/*
 * openchemlib-utils v8.8.0
 * Various utilities that extends openchemlib-js like HOSE codes or diastereotopic IDs
 * https://github.com/cheminfo/openchemlib-utils#readme
 *
 * Licensed under the MIT license.
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.OCLUtils = {}));
})(this, (function (exports) { 'use strict';

    var _documentCurrentScript = typeof document !== 'undefined' ? document.currentScript : null;
    let xAtomicNumber = 0;
    /**
     * Returns the atomic number of the X atom
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @returns
     */
    function getXAtomicNumber(molecule) {
      if (!xAtomicNumber) {
        const OCL = molecule.getOCL();
        xAtomicNumber = OCL.Molecule.getAtomicNoFromLabel('X', OCL.Molecule.cPseudoAtomX);
      }
      return xAtomicNumber;
    }

    /**
     * Check if a specific atom is a sp3 carbon
     * @param {import('openchemlib').Molecule} molecule
     * @param {number} atomID
     */
    function isCsp3(molecule, atomID) {
      if (molecule.getAtomicNo(atomID) !== 6) return false;
      if (molecule.getAtomCharge(atomID) !== 0) return false;
      if (molecule.getImplicitHydrogens(atomID) + molecule.getConnAtoms(atomID) !== 4) {
        return false;
      }
      return true;
    }

    /**
     * This method put all the chiral centers in the molecule in the same group and racemic
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @param {object} [options={}]
     * @param {object} [options.OCL] - openchemlib library
     */
    function makeRacemic(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      // if we don't calculate this we have 2 epimers
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
      // we need to make one group "AND" for chiral (to force to racemic, this means diastereotopic and not enantiotopic)
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (molecule.getAtomParity(i) !== Molecule.cAtomParityNone) {
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0); // changed to group 0; TLS 9.Nov.2015
        }
      }
      // after the change we need to recalculate the CIP
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
    }

    /**
     * Tag an atom to be able to visualize it
     * @param molecule
     * @param iAtom
     */
    function tagAtom(molecule, iAtom) {
      const customLabel = `${molecule.getAtomLabel(iAtom)}*`;
      molecule.setAtomCustomLabel(iAtom, customLabel);
      if (molecule.getAtomicNo(iAtom) === 1) {
        molecule.setAtomicNo(iAtom, getXAtomicNumber(molecule));
      } else {
        // we can not use X because we would have problems with valencies if it is
        // expanded hydrogens or not
        // we can not only use a custom label because it does not count for the canonisation
        molecule.setAtomMass(iAtom, molecule.getAtomMass(iAtom) + 5);
      }
      return customLabel;
    }

    const FULL_HOSE_CODE$1 = 1;
    const HOSE_CODE_CUT_C_SP3_SP3$1 = 2;
    /**
     * Returns an array of hose code fragments for the specified molecule.
     * @param molecule - The OCL molecule to process.
     * @param options - Options for generating hose codes.
     * @returns An array of hose code fragments.
     */
    function getHoseCodesForAtomsAsFragments(molecule, options = {}) {
      const OCL = molecule.getOCL();
      const {
        allowedCustomLabels,
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE$1,
        tagAtoms = [],
        tagAtomFct = tagAtom
      } = options;
      const rootAtoms = options.rootAtoms ? options.rootAtoms.slice() : [];
      molecule = molecule.getCompactCopy();
      if (tagAtoms.length > 0) {
        internalTagAtoms(molecule, tagAtoms, rootAtoms, tagAtomFct);
      } else {
        // this force reordering of atoms in order to have hydrogens at the end
        molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      }
      if (rootAtoms.length === 0) {
        for (let j = 0; j < molecule.getAllAtoms(); j++) {
          if (allowedCustomLabels?.includes(molecule.getAtomCustomLabel(j) ?? '') || molecule.getAtomCustomLabel(j)) {
            rootAtoms.push(j);
          }
        }
      }
      const fragments = [];
      // keep track of the atoms when creating the fragment
      const mappings = [];
      let min = 0;
      let max = 0;
      const atomMask = new Array(molecule.getAllAtoms()).fill(false);
      const atomList = new Array(molecule.getAllAtoms());
      for (let sphere = 0; sphere <= maxSphereSize; sphere++) {
        if (max === 0) {
          for (const rootAtom of rootAtoms) {
            atomList[max] = rootAtom;
            atomMask[rootAtom] = true;
            max++;
          }
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            const atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              const connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                switch (kind) {
                  case FULL_HOSE_CODE$1:
                    atomMask[connAtom] = true;
                    atomList[newMax++] = connAtom;
                    break;
                  case HOSE_CODE_CUT_C_SP3_SP3$1:
                    if (!(isCsp3(molecule, atom) && isCsp3(molecule, connAtom))) {
                      atomMask[connAtom] = true;
                      atomList[newMax++] = connAtom;
                    }
                    break;
                  default:
                    throw new Error('getHoseCoesForAtom unknown kind');
                }
              }
            }
          }
          min = max;
          max = newMax;
        }
        if (sphere >= minSphereSize) {
          const fragment = new OCL.Molecule(0, 0);
          molecule.copyMoleculeByAtoms(fragment, atomMask, true, mappings);
          // we using atomMapNo field in order to keep track of the original atom number even if we remove hydrogens
          for (let i = 0; i < fragment.getAllAtoms(); i++) {
            fragment.setAtomMapNo(i, mappings.indexOf(i) + 1);
          }
          fragment.removeExplicitHydrogens();
          makeRacemic(fragment);
          // we encode atom characteristics in the query features
          addQueryFeaturesAndRemoveMapNo(fragment, molecule);
          fragments.push(fragment);
        }
      }
      return fragments;
    }
    /**
     * If the atom is not an halogen, X or an hydrogen
     * we add query features to the atom
     * This includes aromaticity, ring size, number of hydrogens
     * @param fragment
     * @param molecule
     */
    function addQueryFeaturesAndRemoveMapNo(fragment, molecule) {
      const Molecule = molecule.getOCL().Molecule;
      for (let i = 0; i < fragment.getAllAtoms(); i++) {
        const mapping = fragment.getAtomMapNo(i) - 1;
        fragment.setAtomMapNo(i, 0);
        if ([1, 9, 17, 35, 53, getXAtomicNumber(molecule)].includes(fragment.getAtomicNo(i))) {
          continue;
        }
        // aromaticity
        const isAromatic = molecule.isAromaticAtom(mapping);
        if (isAromatic) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFAromatic, true);
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNotAromatic, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFAromatic, false);
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNotAromatic, true);
        }
        // cycles
        const smallestRing = molecule.getAtomRingSize(mapping);
        switch (smallestRing) {
          case 0:
            break;
          case 3:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize3, true);
            break;
          case 4:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize4, true);
            break;
          case 5:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize5, true);
            break;
          case 6:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize6, true);
            break;
          case 7:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize7, true);
            break;
          default:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSizeLarge, true);
        }
        const nbHydrogens = molecule.getAllHydrogens(mapping);
        if (nbHydrogens === 0) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot0Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot0Hydrogen, true);
        }
        if (nbHydrogens === 1) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot1Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot1Hydrogen, true);
        }
        if (nbHydrogens === 2) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot2Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot2Hydrogen, true);
        }
        if (nbHydrogens === 3) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot3Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot3Hydrogen, true);
        }
      }
    }
    // tagging atoms may change the order of the atoms because hydrogens must be at the end of the file
    // in order to remember the rootAtoms we will tag before
    function internalTagAtoms(molecule, tagAtoms, rootAtoms, tagAtomFct) {
      const OCL = molecule.getOCL();
      if (tagAtoms) {
        for (let i = 0; i < molecule.getAllAtoms(); i++) {
          molecule.setAtomMapNo(i, i + 1);
        }
        if (tagAtoms.length > 0) {
          for (const atom of tagAtoms) {
            tagAtomFct(molecule, atom);
          }
        }
      }
      // this force reordering of atoms in order to have hydrogens at the end
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      if (rootAtoms.length > 0) {
        const mapping = new Int32Array(molecule.getAllAtoms());
        for (let i = 0; i < molecule.getAllAtoms(); i++) {
          mapping[molecule.getAtomMapNo(i) - 1] = i;
        }
        for (let i = 0; i < rootAtoms.length; i++) {
          rootAtoms[i] = mapping[rootAtoms[i]];
        }
      }
    }

    /**
     * We need to create an array of atoms
     * that contains an array of pathLength
     * that contains an array of object
     * @param molecule
     * @param options
     * @returns
     */
    function getAllAtomsPaths(molecule, options = {}) {
      const {
        maxPathLength = 5
      } = options;
      const allAtomsPaths = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        const oneAtomPaths = [];
        allAtomsPaths.push(oneAtomPaths);
        let atomPaths = [];
        atomPaths.push({
          path: [i],
          pathLength: 0
        });
        oneAtomPaths.push(atomPaths);
        let nextIndexes = [0];
        let nextAtoms = [i];
        for (let sphere = 1; sphere <= maxPathLength; sphere++) {
          atomPaths = [];
          oneAtomPaths.push(atomPaths);
          const currentIndexes = nextIndexes;
          const currentAtoms = nextAtoms;
          nextIndexes = [];
          nextAtoms = [];
          for (let i = 0; i < currentIndexes.length; i++) {
            const atom = currentAtoms[i];
            const index = currentIndexes[i];
            const previousPath = oneAtomPaths[sphere - 1]?.[index]?.path;
            if (!previousPath) {
              throw new Error(`Unexpected missing previousPath for sphere ${sphere - 1} and index ${index}`);
            }
            for (let conn = 0; conn < molecule.getAllConnAtoms(atom); conn++) {
              const connectedAtom = molecule.getConnAtom(atom, conn);
              if (previousPath.includes(connectedAtom)) continue;
              nextIndexes.push(atomPaths.length);
              nextAtoms.push(connectedAtom);
              atomPaths.push({
                path: [...previousPath, connectedAtom],
                pathLength: sphere
              });
            }
          }
        }
      }
      return allAtomsPaths;
    }

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    function getAugmentedNamespace(n) {
      if (Object.prototype.hasOwnProperty.call(n, '__esModule')) return n;
      var f = n.default;
    	if (typeof f == "function") {
    		var a = function a () {
    			var isInstance = false;
          try {
            isInstance = this instanceof a;
          } catch {}
    			if (isInstance) {
            return Reflect.construct(f, arguments, this.constructor);
    			}
    			return f.apply(this, arguments);
    		};
    		a.prototype = f.prototype;
      } else a = {};
      Object.defineProperty(a, '__esModule', {value: true});
    	Object.keys(n).forEach(function (k) {
    		var d = Object.getOwnPropertyDescriptor(n, k);
    		Object.defineProperty(a, k, d.get ? d : {
    			enumerable: true,
    			get: function () {
    				return n[k];
    			}
    		});
    	});
    	return a;
    }

    var matrix = {};

    // eslint-disable-next-line @typescript-eslint/unbound-method
    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
     *
     * @param value - Object to check.
     * @returns True if the object is an array or a typed array.
     */
    function isAnyArray$1(value) {
      const tag = toString.call(value);
      return tag.endsWith('Array]') && !tag.includes('Big');
    }

    var libEsm = /*#__PURE__*/Object.freeze({
        __proto__: null,
        isAnyArray: isAnyArray$1
    });

    var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

    function max(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var maxValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] > maxValue) maxValue = input[i];
      }
      return maxValue;
    }

    function min(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var minValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] < minValue) minValue = input[i];
      }
      return minValue;
    }

    function rescale$1(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      } else if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var output;
      if (options.output !== undefined) {
        if (!isAnyArray$1(options.output)) {
          throw new TypeError('output option must be an array if specified');
        }
        output = options.output;
      } else {
        output = new Array(input.length);
      }
      var currentMin = min(input);
      var currentMax = max(input);
      if (currentMin === currentMax) {
        throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
      }
      var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
      if (minValue >= maxValue) {
        throw new RangeError('min option must be smaller than max option');
      }
      var factor = (maxValue - minValue) / (currentMax - currentMin);
      for (var i = 0; i < input.length; i++) {
        output[i] = (input[i] - currentMin) * factor + minValue;
      }
      return output;
    }

    var libEs6 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        default: rescale$1
    });

    var require$$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

    Object.defineProperty(matrix, '__esModule', {
      value: true
    });
    var isAnyArray = require$$0;
    var rescale = require$$1;
    const indent = ' '.repeat(2);
    const indentData = ' '.repeat(4);

    /**
     * @this {Matrix}
     * @returns {string}
     */
    function inspectMatrix() {
      return inspectMatrixWithOptions(this);
    }
    function inspectMatrixWithOptions(matrix, options = {}) {
      const {
        maxRows = 15,
        maxColumns = 10,
        maxNumSize = 8,
        padMinus = 'auto'
      } = options;
      return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
    }
    function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
      const {
        rows,
        columns
      } = matrix;
      const maxI = Math.min(rows, maxRows);
      const maxJ = Math.min(columns, maxColumns);
      const result = [];
      if (padMinus === 'auto') {
        padMinus = false;
        loop: for (let i = 0; i < maxI; i++) {
          for (let j = 0; j < maxJ; j++) {
            if (matrix.get(i, j) < 0) {
              padMinus = true;
              break loop;
            }
          }
        }
      }
      for (let i = 0; i < maxI; i++) {
        let line = [];
        for (let j = 0; j < maxJ; j++) {
          line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
        }
        result.push(`${line.join(' ')}`);
      }
      if (maxJ !== columns) {
        result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
      }
      if (maxI !== rows) {
        result.push(`... ${rows - maxRows} more rows`);
      }
      return result.join(`\n${indentData}`);
    }
    function formatNumber(num, maxNumSize, padMinus) {
      return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
    }
    function formatNumber2(num, len) {
      // small.length numbers should be as is
      let str = num.toString();
      if (str.length <= len) return str;

      // (7)'0.00123' is better then (7)'1.23e-2'
      // (8)'0.000123' is worse then (7)'1.23e-3',
      let fix = num.toFixed(len);
      if (fix.length > len) {
        fix = num.toFixed(Math.max(0, len - (fix.length - len)));
      }
      if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
        return fix;
      }

      // well, if it's still too long the user should've used longer numbers
      let exp = num.toExponential(len);
      if (exp.length > len) {
        exp = num.toExponential(Math.max(0, len - (exp.length - len)));
      }
      return exp.slice(0);
    }
    function installMathOperations(AbstractMatrix, Matrix) {
      AbstractMatrix.prototype.add = function add(value) {
        if (typeof value === 'number') return this.addS(value);
        return this.addM(value);
      };
      AbstractMatrix.prototype.addS = function addS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.addM = function addM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.add = function add(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.add(value);
      };
      AbstractMatrix.prototype.sub = function sub(value) {
        if (typeof value === 'number') return this.subS(value);
        return this.subM(value);
      };
      AbstractMatrix.prototype.subS = function subS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.subM = function subM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.sub = function sub(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sub(value);
      };
      AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
      AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
      AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
      AbstractMatrix.subtract = AbstractMatrix.sub;
      AbstractMatrix.prototype.mul = function mul(value) {
        if (typeof value === 'number') return this.mulS(value);
        return this.mulM(value);
      };
      AbstractMatrix.prototype.mulS = function mulS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.mulM = function mulM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mul = function mul(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mul(value);
      };
      AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
      AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
      AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
      AbstractMatrix.multiply = AbstractMatrix.mul;
      AbstractMatrix.prototype.div = function div(value) {
        if (typeof value === 'number') return this.divS(value);
        return this.divM(value);
      };
      AbstractMatrix.prototype.divS = function divS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.divM = function divM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.div = function div(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.div(value);
      };
      AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
      AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
      AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
      AbstractMatrix.divide = AbstractMatrix.div;
      AbstractMatrix.prototype.mod = function mod(value) {
        if (typeof value === 'number') return this.modS(value);
        return this.modM(value);
      };
      AbstractMatrix.prototype.modS = function modS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.modM = function modM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mod = function mod(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mod(value);
      };
      AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
      AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
      AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
      AbstractMatrix.modulus = AbstractMatrix.mod;
      AbstractMatrix.prototype.and = function and(value) {
        if (typeof value === 'number') return this.andS(value);
        return this.andM(value);
      };
      AbstractMatrix.prototype.andS = function andS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.andM = function andM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.and = function and(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.and(value);
      };
      AbstractMatrix.prototype.or = function or(value) {
        if (typeof value === 'number') return this.orS(value);
        return this.orM(value);
      };
      AbstractMatrix.prototype.orS = function orS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.orM = function orM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.or = function or(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.or(value);
      };
      AbstractMatrix.prototype.xor = function xor(value) {
        if (typeof value === 'number') return this.xorS(value);
        return this.xorM(value);
      };
      AbstractMatrix.prototype.xorS = function xorS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.xorM = function xorM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.xor = function xor(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.xor(value);
      };
      AbstractMatrix.prototype.leftShift = function leftShift(value) {
        if (typeof value === 'number') return this.leftShiftS(value);
        return this.leftShiftM(value);
      };
      AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.leftShift = function leftShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.leftShift(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
        if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
        return this.signPropagatingRightShiftM(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.signPropagatingRightShift(value);
      };
      AbstractMatrix.prototype.rightShift = function rightShift(value) {
        if (typeof value === 'number') return this.rightShiftS(value);
        return this.rightShiftM(value);
      };
      AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.rightShift = function rightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.rightShift(value);
      };
      AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
      AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
      AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
      AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
      AbstractMatrix.prototype.not = function not() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, ~this.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.not = function not(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.not();
      };
      AbstractMatrix.prototype.abs = function abs() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.abs(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.abs = function abs(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.abs();
      };
      AbstractMatrix.prototype.acos = function acos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acos = function acos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acos();
      };
      AbstractMatrix.prototype.acosh = function acosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acosh = function acosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acosh();
      };
      AbstractMatrix.prototype.asin = function asin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asin = function asin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asin();
      };
      AbstractMatrix.prototype.asinh = function asinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asinh = function asinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asinh();
      };
      AbstractMatrix.prototype.atan = function atan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atan = function atan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atan();
      };
      AbstractMatrix.prototype.atanh = function atanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atanh = function atanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atanh();
      };
      AbstractMatrix.prototype.cbrt = function cbrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cbrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cbrt = function cbrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cbrt();
      };
      AbstractMatrix.prototype.ceil = function ceil() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.ceil(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.ceil = function ceil(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.ceil();
      };
      AbstractMatrix.prototype.clz32 = function clz32() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.clz32(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.clz32 = function clz32(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.clz32();
      };
      AbstractMatrix.prototype.cos = function cos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cos = function cos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cos();
      };
      AbstractMatrix.prototype.cosh = function cosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cosh = function cosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cosh();
      };
      AbstractMatrix.prototype.exp = function exp() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.exp(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.exp = function exp(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.exp();
      };
      AbstractMatrix.prototype.expm1 = function expm1() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.expm1(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.expm1 = function expm1(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.expm1();
      };
      AbstractMatrix.prototype.floor = function floor() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.floor(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.floor = function floor(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.floor();
      };
      AbstractMatrix.prototype.fround = function fround() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.fround(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.fround = function fround(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.fround();
      };
      AbstractMatrix.prototype.log = function log() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log = function log(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log();
      };
      AbstractMatrix.prototype.log1p = function log1p() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log1p(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log1p = function log1p(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log1p();
      };
      AbstractMatrix.prototype.log10 = function log10() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log10(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log10 = function log10(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log10();
      };
      AbstractMatrix.prototype.log2 = function log2() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log2(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log2 = function log2(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log2();
      };
      AbstractMatrix.prototype.round = function round() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.round(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.round = function round(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.round();
      };
      AbstractMatrix.prototype.sign = function sign() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sign(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sign = function sign(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sign();
      };
      AbstractMatrix.prototype.sin = function sin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sin = function sin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sin();
      };
      AbstractMatrix.prototype.sinh = function sinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sinh = function sinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sinh();
      };
      AbstractMatrix.prototype.sqrt = function sqrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sqrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sqrt = function sqrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sqrt();
      };
      AbstractMatrix.prototype.tan = function tan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tan = function tan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tan();
      };
      AbstractMatrix.prototype.tanh = function tanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tanh = function tanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tanh();
      };
      AbstractMatrix.prototype.trunc = function trunc() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.trunc(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.trunc = function trunc(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.trunc();
      };
      AbstractMatrix.pow = function pow(matrix, arg0) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.pow(arg0);
      };
      AbstractMatrix.prototype.pow = function pow(value) {
        if (typeof value === 'number') return this.powS(value);
        return this.powM(value);
      };
      AbstractMatrix.prototype.powS = function powS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.powM = function powM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** matrix.get(i, j));
          }
        }
        return this;
      };
    }

    /**
     * @private
     * Check that a row index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkRowIndex(matrix, index, outer) {
      let max = outer ? matrix.rows : matrix.rows - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Row index out of range');
      }
    }

    /**
     * @private
     * Check that a column index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkColumnIndex(matrix, index, outer) {
      let max = outer ? matrix.columns : matrix.columns - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Column index out of range');
      }
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkRowVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.columns) {
        throw new RangeError('vector size must be the same as the number of columns');
      }
      return vector;
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkColumnVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.rows) {
        throw new RangeError('vector size must be the same as the number of rows');
      }
      return vector;
    }
    function checkRowIndices(matrix, rowIndices) {
      if (!isAnyArray.isAnyArray(rowIndices)) {
        throw new TypeError('row indices must be an array');
      }
      for (let i = 0; i < rowIndices.length; i++) {
        if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
          throw new RangeError('row indices are out of range');
        }
      }
    }
    function checkColumnIndices(matrix, columnIndices) {
      if (!isAnyArray.isAnyArray(columnIndices)) {
        throw new TypeError('column indices must be an array');
      }
      for (let i = 0; i < columnIndices.length; i++) {
        if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
          throw new RangeError('column indices are out of range');
        }
      }
    }
    function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
      if (arguments.length !== 5) {
        throw new RangeError('expected 4 arguments');
      }
      checkNumber('startRow', startRow);
      checkNumber('endRow', endRow);
      checkNumber('startColumn', startColumn);
      checkNumber('endColumn', endColumn);
      if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
        throw new RangeError('Submatrix indices are out of range');
      }
    }
    function newArray(length, value = 0) {
      let array = [];
      for (let i = 0; i < length; i++) {
        array.push(value);
      }
      return array;
    }
    function checkNumber(name, value) {
      if (typeof value !== 'number') {
        throw new TypeError(`${name} must be a number`);
      }
    }
    function checkNonEmpty(matrix) {
      if (matrix.isEmpty()) {
        throw new Error('Empty matrix has no elements to index');
      }
    }
    function sumByRow(matrix) {
      let sum = newArray(matrix.rows);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumByColumn(matrix) {
      let sum = newArray(matrix.columns);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumAll(matrix) {
      let v = 0;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v += matrix.get(i, j);
        }
      }
      return v;
    }
    function productByRow(matrix) {
      let sum = newArray(matrix.rows, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productByColumn(matrix) {
      let sum = newArray(matrix.columns, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productAll(matrix) {
      let v = 1;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v *= matrix.get(i, j);
        }
      }
      return v;
    }
    function varianceByRow(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let i = 0; i < rows; i++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean[i];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / cols) / cols);
        }
      }
      return variance;
    }
    function varianceByColumn(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let j = 0; j < cols; j++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let i = 0; i < rows; i++) {
          x = matrix.get(i, j) - mean[j];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / rows) / rows);
        }
      }
      return variance;
    }
    function varianceAll(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const size = rows * cols;
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean;
          sum1 += x;
          sum2 += x * x;
        }
      }
      if (unbiased) {
        return (sum2 - sum1 * sum1 / size) / (size - 1);
      } else {
        return (sum2 - sum1 * sum1 / size) / size;
      }
    }
    function centerByRow(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[i]);
        }
      }
    }
    function centerByColumn(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[j]);
        }
      }
    }
    function centerAll(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean);
        }
      }
    }
    function getScaleByRow(matrix) {
      const scale = [];
      for (let i = 0; i < matrix.rows; i++) {
        let sum = 0;
        for (let j = 0; j < matrix.columns; j++) {
          sum += matrix.get(i, j) ** 2 / (matrix.columns - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByRow(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[i]);
        }
      }
    }
    function getScaleByColumn(matrix) {
      const scale = [];
      for (let j = 0; j < matrix.columns; j++) {
        let sum = 0;
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / (matrix.rows - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByColumn(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[j]);
        }
      }
    }
    function getScaleAll(matrix) {
      const divider = matrix.size - 1;
      let sum = 0;
      for (let j = 0; j < matrix.columns; j++) {
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / divider;
        }
      }
      return Math.sqrt(sum);
    }
    function scaleAll(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale);
        }
      }
    }
    class AbstractMatrix {
      static from1DArray(newRows, newColumns, newData) {
        let length = newRows * newColumns;
        if (length !== newData.length) {
          throw new RangeError('data length does not match given dimensions');
        }
        let newMatrix = new Matrix$1(newRows, newColumns);
        for (let row = 0; row < newRows; row++) {
          for (let column = 0; column < newColumns; column++) {
            newMatrix.set(row, column, newData[row * newColumns + column]);
          }
        }
        return newMatrix;
      }
      static rowVector(newData) {
        let vector = new Matrix$1(1, newData.length);
        for (let i = 0; i < newData.length; i++) {
          vector.set(0, i, newData[i]);
        }
        return vector;
      }
      static columnVector(newData) {
        let vector = new Matrix$1(newData.length, 1);
        for (let i = 0; i < newData.length; i++) {
          vector.set(i, 0, newData[i]);
        }
        return vector;
      }
      static zeros(rows, columns) {
        return new Matrix$1(rows, columns);
      }
      static ones(rows, columns) {
        return new Matrix$1(rows, columns).fill(1);
      }
      static rand(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          random = Math.random
        } = options;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.set(i, j, random());
          }
        }
        return matrix;
      }
      static randInt(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1000,
          random = Math.random
        } = options;
        if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
        if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let interval = max - min;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            let value = min + Math.round(random() * interval);
            matrix.set(i, j, value);
          }
        }
        return matrix;
      }
      static eye(rows, columns, value) {
        if (columns === undefined) columns = rows;
        if (value === undefined) value = 1;
        let min = Math.min(rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, value);
        }
        return matrix;
      }
      static diag(data, rows, columns) {
        let l = data.length;
        if (rows === undefined) rows = l;
        if (columns === undefined) columns = rows;
        let min = Math.min(l, rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, data[i]);
        }
        return matrix;
      }
      static min(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static max(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new this(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static checkMatrix(value) {
        return AbstractMatrix.isMatrix(value) ? value : new Matrix$1(value);
      }
      static isMatrix(value) {
        return value != null && value.klass === 'Matrix';
      }
      get size() {
        return this.rows * this.columns;
      }
      apply(callback) {
        if (typeof callback !== 'function') {
          throw new TypeError('callback must be a function');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            callback.call(this, i, j);
          }
        }
        return this;
      }
      to1DArray() {
        let array = [];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            array.push(this.get(i, j));
          }
        }
        return array;
      }
      to2DArray() {
        let copy = [];
        for (let i = 0; i < this.rows; i++) {
          copy.push([]);
          for (let j = 0; j < this.columns; j++) {
            copy[i].push(this.get(i, j));
          }
        }
        return copy;
      }
      toJSON() {
        return this.to2DArray();
      }
      isRowVector() {
        return this.rows === 1;
      }
      isColumnVector() {
        return this.columns === 1;
      }
      isVector() {
        return this.rows === 1 || this.columns === 1;
      }
      isSquare() {
        return this.rows === this.columns;
      }
      isEmpty() {
        return this.rows === 0 || this.columns === 0;
      }
      isSymmetric() {
        if (this.isSquare()) {
          for (let i = 0; i < this.rows; i++) {
            for (let j = 0; j <= i; j++) {
              if (this.get(i, j) !== this.get(j, i)) {
                return false;
              }
            }
          }
          return true;
        }
        return false;
      }
      isDistance() {
        if (!this.isSymmetric()) return false;
        for (let i = 0; i < this.rows; i++) {
          if (this.get(i, i) !== 0) return false;
        }
        return true;
      }
      isEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isEchelonForm = true;
        let checked = false;
        while (i < this.rows && isEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isEchelonForm = false;
              checked = true;
            }
          }
          i++;
        }
        return isEchelonForm;
      }
      isReducedEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isReducedEchelonForm = true;
        let checked = false;
        while (i < this.rows && isReducedEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isReducedEchelonForm = false;
              checked = true;
            }
          }
          for (let k = j + 1; k < this.rows; k++) {
            if (this.get(i, k) !== 0) {
              isReducedEchelonForm = false;
            }
          }
          i++;
        }
        return isReducedEchelonForm;
      }
      echelonForm() {
        let result = this.clone();
        let h = 0;
        let k = 0;
        while (h < result.rows && k < result.columns) {
          let iMax = h;
          for (let i = h; i < result.rows; i++) {
            if (result.get(i, k) > result.get(iMax, k)) {
              iMax = i;
            }
          }
          if (result.get(iMax, k) === 0) {
            k++;
          } else {
            result.swapRows(h, iMax);
            let tmp = result.get(h, k);
            for (let j = k; j < result.columns; j++) {
              result.set(h, j, result.get(h, j) / tmp);
            }
            for (let i = h + 1; i < result.rows; i++) {
              let factor = result.get(i, k) / result.get(h, k);
              result.set(i, k, 0);
              for (let j = k + 1; j < result.columns; j++) {
                result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
              }
            }
            h++;
            k++;
          }
        }
        return result;
      }
      reducedEchelonForm() {
        let result = this.echelonForm();
        let m = result.columns;
        let n = result.rows;
        let h = n - 1;
        while (h >= 0) {
          if (result.maxRow(h) === 0) {
            h--;
          } else {
            let p = 0;
            let pivot = false;
            while (p < n && pivot === false) {
              if (result.get(h, p) === 1) {
                pivot = true;
              } else {
                p++;
              }
            }
            for (let i = 0; i < h; i++) {
              let factor = result.get(i, p);
              for (let j = p; j < m; j++) {
                let tmp = result.get(i, j) - factor * result.get(h, j);
                result.set(i, j, tmp);
              }
            }
            h--;
          }
        }
        return result;
      }
      set() {
        throw new Error('set method is unimplemented');
      }
      get() {
        throw new Error('get method is unimplemented');
      }
      repeat(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          rows = 1,
          columns = 1
        } = options;
        if (!Number.isInteger(rows) || rows <= 0) {
          throw new TypeError('rows must be a positive integer');
        }
        if (!Number.isInteger(columns) || columns <= 0) {
          throw new TypeError('columns must be a positive integer');
        }
        let matrix = new Matrix$1(this.rows * rows, this.columns * columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.setSubMatrix(this, this.rows * i, this.columns * j);
          }
        }
        return matrix;
      }
      fill(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, value);
          }
        }
        return this;
      }
      neg() {
        return this.mulS(-1);
      }
      getRow(index) {
        checkRowIndex(this, index);
        let row = [];
        for (let i = 0; i < this.columns; i++) {
          row.push(this.get(index, i));
        }
        return row;
      }
      getRowVector(index) {
        return Matrix$1.rowVector(this.getRow(index));
      }
      setRow(index, array) {
        checkRowIndex(this, index);
        array = checkRowVector(this, array);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, array[i]);
        }
        return this;
      }
      swapRows(row1, row2) {
        checkRowIndex(this, row1);
        checkRowIndex(this, row2);
        for (let i = 0; i < this.columns; i++) {
          let temp = this.get(row1, i);
          this.set(row1, i, this.get(row2, i));
          this.set(row2, i, temp);
        }
        return this;
      }
      getColumn(index) {
        checkColumnIndex(this, index);
        let column = [];
        for (let i = 0; i < this.rows; i++) {
          column.push(this.get(i, index));
        }
        return column;
      }
      getColumnVector(index) {
        return Matrix$1.columnVector(this.getColumn(index));
      }
      setColumn(index, array) {
        checkColumnIndex(this, index);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, array[i]);
        }
        return this;
      }
      swapColumns(column1, column2) {
        checkColumnIndex(this, column1);
        checkColumnIndex(this, column2);
        for (let i = 0; i < this.rows; i++) {
          let temp = this.get(i, column1);
          this.set(i, column1, this.get(i, column2));
          this.set(i, column2, temp);
        }
        return this;
      }
      addRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[j]);
          }
        }
        return this;
      }
      subRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[j]);
          }
        }
        return this;
      }
      mulRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[j]);
          }
        }
        return this;
      }
      divRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[j]);
          }
        }
        return this;
      }
      addColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[i]);
          }
        }
        return this;
      }
      subColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[i]);
          }
        }
        return this;
      }
      mulColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[i]);
          }
        }
        return this;
      }
      divColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[i]);
          }
        }
        return this;
      }
      mulRow(index, value) {
        checkRowIndex(this, index);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, this.get(index, i) * value);
        }
        return this;
      }
      mulColumn(index, value) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, this.get(i, index) * value);
        }
        return this;
      }
      max(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[row]) {
                    max[row] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case 'column':
            {
              const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[column]) {
                    max[column] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case undefined:
            {
              let max = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max) {
                    max = this.get(row, column);
                  }
                }
              }
              return max;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      maxIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) > v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      min(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[row]) {
                    min[row] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case 'column':
            {
              const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[column]) {
                    min[column] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case undefined:
            {
              let min = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min) {
                    min = this.get(row, column);
                  }
                }
              }
              return min;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      minIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) < v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      maxRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      maxRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      minRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      minRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      maxColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      maxColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      minColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      minColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      diag() {
        let min = Math.min(this.rows, this.columns);
        let diag = [];
        for (let i = 0; i < min; i++) {
          diag.push(this.get(i, i));
        }
        return diag;
      }
      norm(type = 'frobenius') {
        switch (type) {
          case 'max':
            return this.max();
          case 'frobenius':
            return Math.sqrt(this.dot(this));
          default:
            throw new RangeError(`unknown norm type: ${type}`);
        }
      }
      cumulativeSum() {
        let sum = 0;
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            sum += this.get(i, j);
            this.set(i, j, sum);
          }
        }
        return this;
      }
      dot(vector2) {
        if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
        let vector1 = this.to1DArray();
        if (vector1.length !== vector2.length) {
          throw new RangeError('vectors do not have the same size');
        }
        let dot = 0;
        for (let i = 0; i < vector1.length; i++) {
          dot += vector1[i] * vector2[i];
        }
        return dot;
      }
      mmul(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.columns;
        let result = new Matrix$1(m, p);
        let Bcolj = new Float64Array(n);
        for (let j = 0; j < p; j++) {
          for (let k = 0; k < n; k++) {
            Bcolj[k] = other.get(k, j);
          }
          for (let i = 0; i < m; i++) {
            let s = 0;
            for (let k = 0; k < n; k++) {
              s += this.get(i, k) * Bcolj[k];
            }
            result.set(i, j, s);
          }
        }
        return result;
      }
      mpow(scalar) {
        if (!this.isSquare()) {
          throw new RangeError('Matrix must be square');
        }
        if (!Number.isInteger(scalar) || scalar < 0) {
          throw new RangeError('Exponent must be a non-negative integer');
        }
        // Russian Peasant exponentiation, i.e. exponentiation by squaring
        let result = Matrix$1.eye(this.rows);
        let bb = this;
        // Note: Don't bit shift. In JS, that would truncate at 32 bits
        for (let e = scalar; e >= 1; e /= 2) {
          if ((e & 1) !== 0) {
            result = result.mmul(bb);
          }
          bb = bb.mmul(bb);
        }
        return result;
      }
      strassen2x2(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(2, 2);
        const a11 = this.get(0, 0);
        const b11 = other.get(0, 0);
        const a12 = this.get(0, 1);
        const b12 = other.get(0, 1);
        const a21 = this.get(1, 0);
        const b21 = other.get(1, 0);
        const a22 = this.get(1, 1);
        const b22 = other.get(1, 1);

        // Compute intermediate values.
        const m1 = (a11 + a22) * (b11 + b22);
        const m2 = (a21 + a22) * b11;
        const m3 = a11 * (b12 - b22);
        const m4 = a22 * (b21 - b11);
        const m5 = (a11 + a12) * b22;
        const m6 = (a21 - a11) * (b11 + b12);
        const m7 = (a12 - a22) * (b21 + b22);

        // Combine intermediate values into the output.
        const c00 = m1 + m4 - m5 + m7;
        const c01 = m3 + m5;
        const c10 = m2 + m4;
        const c11 = m1 - m2 + m3 + m6;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        return result;
      }
      strassen3x3(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(3, 3);
        const a00 = this.get(0, 0);
        const a01 = this.get(0, 1);
        const a02 = this.get(0, 2);
        const a10 = this.get(1, 0);
        const a11 = this.get(1, 1);
        const a12 = this.get(1, 2);
        const a20 = this.get(2, 0);
        const a21 = this.get(2, 1);
        const a22 = this.get(2, 2);
        const b00 = other.get(0, 0);
        const b01 = other.get(0, 1);
        const b02 = other.get(0, 2);
        const b10 = other.get(1, 0);
        const b11 = other.get(1, 1);
        const b12 = other.get(1, 2);
        const b20 = other.get(2, 0);
        const b21 = other.get(2, 1);
        const b22 = other.get(2, 2);
        const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
        const m2 = (a00 - a10) * (-b01 + b11);
        const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
        const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
        const m5 = (a10 + a11) * (-b00 + b01);
        const m6 = a00 * b00;
        const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
        const m8 = (-a00 + a20) * (b02 - b12);
        const m9 = (a20 + a21) * (-b00 + b02);
        const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
        const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
        const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
        const m13 = (a02 - a22) * (b11 - b21);
        const m14 = a02 * b20;
        const m15 = (a21 + a22) * (-b20 + b21);
        const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
        const m17 = (a02 - a12) * (b12 - b22);
        const m18 = (a11 + a12) * (-b20 + b22);
        const m19 = a01 * b10;
        const m20 = a12 * b21;
        const m21 = a10 * b02;
        const m22 = a20 * b01;
        const m23 = a22 * b22;
        const c00 = m6 + m14 + m19;
        const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
        const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
        const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
        const c11 = m2 + m4 + m5 + m6 + m20;
        const c12 = m14 + m16 + m17 + m18 + m21;
        const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
        const c21 = m12 + m13 + m14 + m15 + m22;
        const c22 = m6 + m7 + m8 + m9 + m23;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(0, 2, c02);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        result.set(1, 2, c12);
        result.set(2, 0, c20);
        result.set(2, 1, c21);
        result.set(2, 2, c22);
        return result;
      }
      mmulStrassen(y) {
        y = Matrix$1.checkMatrix(y);
        let x = this.clone();
        let r1 = x.rows;
        let c1 = x.columns;
        let r2 = y.rows;
        let c2 = y.columns;
        if (c1 !== r2) {
          // eslint-disable-next-line no-console
          console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
        }

        // Put a matrix into the top left of a matrix of zeros.
        // `rows` and `cols` are the dimensions of the output matrix.
        function embed(mat, rows, cols) {
          let r = mat.rows;
          let c = mat.columns;
          if (r === rows && c === cols) {
            return mat;
          } else {
            let resultat = AbstractMatrix.zeros(rows, cols);
            resultat = resultat.setSubMatrix(mat, 0, 0);
            return resultat;
          }
        }

        // Make sure both matrices are the same size.
        // This is exclusively for simplicity:
        // this algorithm can be implemented with matrices of different sizes.

        let r = Math.max(r1, r2);
        let c = Math.max(c1, c2);
        x = embed(x, r, c);
        y = embed(y, r, c);

        // Our recursive multiplication function.
        function blockMult(a, b, rows, cols) {
          // For small matrices, resort to naive multiplication.
          if (rows <= 512 || cols <= 512) {
            return a.mmul(b); // a is equivalent to this
          }

          // Apply dynamic padding.
          if (rows % 2 === 1 && cols % 2 === 1) {
            a = embed(a, rows + 1, cols + 1);
            b = embed(b, rows + 1, cols + 1);
          } else if (rows % 2 === 1) {
            a = embed(a, rows + 1, cols);
            b = embed(b, rows + 1, cols);
          } else if (cols % 2 === 1) {
            a = embed(a, rows, cols + 1);
            b = embed(b, rows, cols + 1);
          }
          let halfRows = parseInt(a.rows / 2, 10);
          let halfCols = parseInt(a.columns / 2, 10);
          // Subdivide input matrices.
          let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
          let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
          let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
          let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
          let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
          let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

          // Compute intermediate values.
          let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
          let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
          let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
          let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
          let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
          let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
          let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

          // Combine intermediate values into the output.
          let c11 = AbstractMatrix.add(m1, m4);
          c11.sub(m5);
          c11.add(m7);
          let c12 = AbstractMatrix.add(m3, m5);
          let c21 = AbstractMatrix.add(m2, m4);
          let c22 = AbstractMatrix.sub(m1, m2);
          c22.add(m3);
          c22.add(m6);

          // Crop output to the desired size (undo dynamic padding).
          let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
          result = result.setSubMatrix(c11, 0, 0);
          result = result.setSubMatrix(c12, c11.rows, 0);
          result = result.setSubMatrix(c21, 0, c11.columns);
          result = result.setSubMatrix(c22, c11.rows, c11.columns);
          return result.subMatrix(0, rows - 1, 0, cols - 1);
        }
        return blockMult(x, y, r, c);
      }
      scaleRows(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.rows; i++) {
          const row = this.getRow(i);
          if (row.length > 0) {
            rescale(row, {
              min,
              max,
              output: row
            });
          }
          newMatrix.setRow(i, row);
        }
        return newMatrix;
      }
      scaleColumns(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.columns; i++) {
          const column = this.getColumn(i);
          if (column.length) {
            rescale(column, {
              min,
              max,
              output: column
            });
          }
          newMatrix.setColumn(i, column);
        }
        return newMatrix;
      }
      flipRows() {
        const middle = Math.ceil(this.columns / 2);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < middle; j++) {
            let first = this.get(i, j);
            let last = this.get(i, this.columns - 1 - j);
            this.set(i, j, last);
            this.set(i, this.columns - 1 - j, first);
          }
        }
        return this;
      }
      flipColumns() {
        const middle = Math.ceil(this.rows / 2);
        for (let j = 0; j < this.columns; j++) {
          for (let i = 0; i < middle; i++) {
            let first = this.get(i, j);
            let last = this.get(this.rows - 1 - i, j);
            this.set(i, j, last);
            this.set(this.rows - 1 - i, j, first);
          }
        }
        return this;
      }
      kroneckerProduct(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.rows;
        let q = other.columns;
        let result = new Matrix$1(m * p, n * q);
        for (let i = 0; i < m; i++) {
          for (let j = 0; j < n; j++) {
            for (let k = 0; k < p; k++) {
              for (let l = 0; l < q; l++) {
                result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
              }
            }
          }
        }
        return result;
      }
      kroneckerSum(other) {
        other = Matrix$1.checkMatrix(other);
        if (!this.isSquare() || !other.isSquare()) {
          throw new Error('Kronecker Sum needs two Square Matrices');
        }
        let m = this.rows;
        let n = other.rows;
        let AxI = this.kroneckerProduct(Matrix$1.eye(n, n));
        let IxB = Matrix$1.eye(m, m).kroneckerProduct(other);
        return AxI.add(IxB);
      }
      transpose() {
        let result = new Matrix$1(this.columns, this.rows);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result.set(j, i, this.get(i, j));
          }
        }
        return result;
      }
      sortRows(compareFunction = compareNumbers) {
        for (let i = 0; i < this.rows; i++) {
          this.setRow(i, this.getRow(i).sort(compareFunction));
        }
        return this;
      }
      sortColumns(compareFunction = compareNumbers) {
        for (let i = 0; i < this.columns; i++) {
          this.setColumn(i, this.getColumn(i).sort(compareFunction));
        }
        return this;
      }
      subMatrix(startRow, endRow, startColumn, endColumn) {
        checkRange(this, startRow, endRow, startColumn, endColumn);
        let newMatrix = new Matrix$1(endRow - startRow + 1, endColumn - startColumn + 1);
        for (let i = startRow; i <= endRow; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
          }
        }
        return newMatrix;
      }
      subMatrixRow(indices, startColumn, endColumn) {
        if (startColumn === undefined) startColumn = 0;
        if (endColumn === undefined) endColumn = this.columns - 1;
        if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(indices.length, endColumn - startColumn + 1);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            if (indices[i] < 0 || indices[i] >= this.rows) {
              throw new RangeError(`Row index out of range: ${indices[i]}`);
            }
            newMatrix.set(i, j - startColumn, this.get(indices[i], j));
          }
        }
        return newMatrix;
      }
      subMatrixColumn(indices, startRow, endRow) {
        if (startRow === undefined) startRow = 0;
        if (endRow === undefined) endRow = this.rows - 1;
        if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(endRow - startRow + 1, indices.length);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startRow; j <= endRow; j++) {
            if (indices[i] < 0 || indices[i] >= this.columns) {
              throw new RangeError(`Column index out of range: ${indices[i]}`);
            }
            newMatrix.set(j - startRow, i, this.get(j, indices[i]));
          }
        }
        return newMatrix;
      }
      setSubMatrix(matrix, startRow, startColumn) {
        matrix = Matrix$1.checkMatrix(matrix);
        if (matrix.isEmpty()) {
          return this;
        }
        let endRow = startRow + matrix.rows - 1;
        let endColumn = startColumn + matrix.columns - 1;
        checkRange(this, startRow, endRow, startColumn, endColumn);
        for (let i = 0; i < matrix.rows; i++) {
          for (let j = 0; j < matrix.columns; j++) {
            this.set(startRow + i, startColumn + j, matrix.get(i, j));
          }
        }
        return this;
      }
      selection(rowIndices, columnIndices) {
        checkRowIndices(this, rowIndices);
        checkColumnIndices(this, columnIndices);
        let newMatrix = new Matrix$1(rowIndices.length, columnIndices.length);
        for (let i = 0; i < rowIndices.length; i++) {
          let rowIndex = rowIndices[i];
          for (let j = 0; j < columnIndices.length; j++) {
            let columnIndex = columnIndices[j];
            newMatrix.set(i, j, this.get(rowIndex, columnIndex));
          }
        }
        return newMatrix;
      }
      trace() {
        let min = Math.min(this.rows, this.columns);
        let trace = 0;
        for (let i = 0; i < min; i++) {
          trace += this.get(i, i);
        }
        return trace;
      }
      clone() {
        return this.constructor.copy(this, new Matrix$1(this.rows, this.columns));
      }

      /**
       * @template {AbstractMatrix} M
       * @param {AbstractMatrix} from
       * @param {M} to
       * @return {M}
       */
      static copy(from, to) {
        for (const [row, column, value] of from.entries()) {
          to.set(row, column, value);
        }
        return to;
      }
      sum(by) {
        switch (by) {
          case 'row':
            return sumByRow(this);
          case 'column':
            return sumByColumn(this);
          case undefined:
            return sumAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      product(by) {
        switch (by) {
          case 'row':
            return productByRow(this);
          case 'column':
            return productByColumn(this);
          case undefined:
            return productAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      mean(by) {
        const sum = this.sum(by);
        switch (by) {
          case 'row':
            {
              for (let i = 0; i < this.rows; i++) {
                sum[i] /= this.columns;
              }
              return sum;
            }
          case 'column':
            {
              for (let i = 0; i < this.columns; i++) {
                sum[i] /= this.rows;
              }
              return sum;
            }
          case undefined:
            return sum / this.size;
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      variance(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          unbiased = true,
          mean = this.mean(by)
        } = options;
        if (typeof unbiased !== 'boolean') {
          throw new TypeError('unbiased must be a boolean');
        }
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByRow(this, unbiased, mean);
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByColumn(this, unbiased, mean);
            }
          case undefined:
            {
              if (typeof mean !== 'number') {
                throw new TypeError('mean must be a number');
              }
              return varianceAll(this, unbiased, mean);
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      standardDeviation(by, options) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        const variance = this.variance(by, options);
        if (by === undefined) {
          return Math.sqrt(variance);
        } else {
          for (let i = 0; i < variance.length; i++) {
            variance[i] = Math.sqrt(variance[i]);
          }
          return variance;
        }
      }
      center(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          center = this.mean(by)
        } = options;
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByRow(this, center);
              return this;
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByColumn(this, center);
              return this;
            }
          case undefined:
            {
              if (typeof center !== 'number') {
                throw new TypeError('center must be a number');
              }
              centerAll(this, center);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      scale(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        let scale = options.scale;
        switch (by) {
          case 'row':
            {
              if (scale === undefined) {
                scale = getScaleByRow(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByRow(this, scale);
              return this;
            }
          case 'column':
            {
              if (scale === undefined) {
                scale = getScaleByColumn(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByColumn(this, scale);
              return this;
            }
          case undefined:
            {
              if (scale === undefined) {
                scale = getScaleAll(this);
              } else if (typeof scale !== 'number') {
                throw new TypeError('scale must be a number');
              }
              scaleAll(this, scale);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      toString(options) {
        return inspectMatrixWithOptions(this, options);
      }
      [Symbol.iterator]() {
        return this.entries();
      }

      /**
       * iterator from left to right, from top to bottom
       * yield [row, column, value]
       * @returns {Generator<[number, number, number], void, void>}
       */
      *entries() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield [row, col, this.get(row, col)];
          }
        }
      }

      /**
       * iterator from left to right, from top to bottom
       * yield value
       * @returns {Generator<number, void, void>}
       */
      *values() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield this.get(row, col);
          }
        }
      }
    }
    AbstractMatrix.prototype.klass = 'Matrix';
    if (typeof Symbol !== 'undefined') {
      AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
    }
    function compareNumbers(a, b) {
      return a - b;
    }
    function isArrayOfNumbers(array) {
      return array.every(element => {
        return typeof element === 'number';
      });
    }

    // Synonyms
    AbstractMatrix.random = AbstractMatrix.rand;
    AbstractMatrix.randomInt = AbstractMatrix.randInt;
    AbstractMatrix.diagonal = AbstractMatrix.diag;
    AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
    AbstractMatrix.identity = AbstractMatrix.eye;
    AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
    AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
    let Matrix$1 = class Matrix extends AbstractMatrix {
      /**
       * @type {Float64Array[]}
       */
      data;

      /**
       * Init an empty matrix
       * @param {number} nRows
       * @param {number} nColumns
       */
      #initData(nRows, nColumns) {
        this.data = [];
        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
        this.rows = nRows;
        this.columns = nColumns;
      }
      constructor(nRows, nColumns) {
        super();
        if (Matrix.isMatrix(nRows)) {
          this.#initData(nRows.rows, nRows.columns);
          Matrix.copy(nRows, this);
        } else if (Number.isInteger(nRows) && nRows >= 0) {
          this.#initData(nRows, nColumns);
        } else if (isAnyArray.isAnyArray(nRows)) {
          // Copy the values from the 2D array
          const arrayData = nRows;
          nRows = arrayData.length;
          nColumns = nRows ? arrayData[0].length : 0;
          if (typeof nColumns !== 'number') {
            throw new TypeError('Data must be a 2D array with at least one element');
          }
          this.data = [];
          for (let i = 0; i < nRows; i++) {
            if (arrayData[i].length !== nColumns) {
              throw new RangeError('Inconsistent array dimensions');
            }
            if (!isArrayOfNumbers(arrayData[i])) {
              throw new TypeError('Input data contains non-numeric values');
            }
            this.data.push(Float64Array.from(arrayData[i]));
          }
          this.rows = nRows;
          this.columns = nColumns;
        } else {
          throw new TypeError('First argument must be a positive number or an array');
        }
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
      removeRow(index) {
        checkRowIndex(this, index);
        this.data.splice(index, 1);
        this.rows -= 1;
        return this;
      }
      addRow(index, array) {
        if (array === undefined) {
          array = index;
          index = this.rows;
        }
        checkRowIndex(this, index, true);
        array = Float64Array.from(checkRowVector(this, array));
        this.data.splice(index, 0, array);
        this.rows += 1;
        return this;
      }
      removeColumn(index) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns - 1);
          for (let j = 0; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          for (let j = index + 1; j < this.columns; j++) {
            newRow[j - 1] = this.data[i][j];
          }
          this.data[i] = newRow;
        }
        this.columns -= 1;
        return this;
      }
      addColumn(index, array) {
        if (typeof array === 'undefined') {
          array = index;
          index = this.columns;
        }
        checkColumnIndex(this, index, true);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns + 1);
          let j = 0;
          for (; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          newRow[j++] = array[i];
          for (; j < this.columns + 1; j++) {
            newRow[j] = this.data[i][j - 1];
          }
          this.data[i] = newRow;
        }
        this.columns += 1;
        return this;
      }
    };
    installMathOperations(AbstractMatrix, Matrix$1);

    /**
     * @typedef {0 | 1 | number | boolean} Mask
     */

    class SymmetricMatrix extends AbstractMatrix {
      /** @type {Matrix} */
      #matrix;
      get size() {
        return this.#matrix.size;
      }
      get rows() {
        return this.#matrix.rows;
      }
      get columns() {
        return this.#matrix.columns;
      }
      get diagonalSize() {
        return this.rows;
      }

      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isSymmetricMatrix(value) {
        return Matrix$1.isMatrix(value) && value.klassType === 'SymmetricMatrix';
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static zeros(diagonalSize) {
        return new this(diagonalSize);
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static ones(diagonalSize) {
        return new this(diagonalSize).fill(1);
      }

      /**
       * @param {number | AbstractMatrix | ArrayLike<ArrayLike<number>>} diagonalSize
       * @return {this}
       */
      constructor(diagonalSize) {
        super();
        if (Matrix$1.isMatrix(diagonalSize)) {
          if (!diagonalSize.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
          this.#matrix = Matrix$1.copy(diagonalSize, new Matrix$1(diagonalSize.rows, diagonalSize.rows));
        } else if (Number.isInteger(diagonalSize) && diagonalSize >= 0) {
          this.#matrix = new Matrix$1(diagonalSize, diagonalSize);
        } else {
          this.#matrix = new Matrix$1(diagonalSize);
          if (!this.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
        }
      }
      clone() {
        const matrix = new SymmetricMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          matrix.set(row, col, value);
        }
        return matrix;
      }
      toMatrix() {
        return new Matrix$1(this);
      }
      get(rowIndex, columnIndex) {
        return this.#matrix.get(rowIndex, columnIndex);
      }
      set(rowIndex, columnIndex, value) {
        // symmetric set
        this.#matrix.set(rowIndex, columnIndex, value);
        this.#matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      removeCross(index) {
        // symmetric remove side
        this.#matrix.removeRow(index);
        this.#matrix.removeColumn(index);
        return this;
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }
        const row = array.slice();
        row.splice(index, 1);
        this.#matrix.addRow(index, row);
        this.#matrix.addColumn(index, array);
        return this;
      }

      /**
       * @param {Mask[]} mask
       */
      applyMask(mask) {
        if (mask.length !== this.diagonalSize) {
          throw new RangeError('Mask size do not match with matrix size');
        }

        // prepare sides to remove from matrix from mask
        /** @type {number[]} */
        const sidesToRemove = [];
        for (const [index, passthroughs] of mask.entries()) {
          if (passthroughs) continue;
          sidesToRemove.push(index);
        }
        // to remove from highest to lowest for no mutation shifting
        sidesToRemove.reverse();

        // remove sides
        for (const sideIndex of sidesToRemove) {
          this.removeCross(sideIndex);
        }
        return this;
      }

      /**
       * Compact format upper-right corner of matrix
       * iterate from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 1 2 3 4
       * B 2 5 6 7
       * C 3 6 8 9
       * D 4 7 9 10
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]`
       *
       * length is S(i=0, n=sideSize) => 10 for a 4 sideSized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;

        /** @type {number[]} */
        const compact = new Array(diagonalSize * (diagonalSize + 1) / 2);
        for (let col = 0, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       * @return {SymmetricMatrix}
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        // compactSize = (sideSize * (sideSize + 1)) / 2
        // https://mathsolver.microsoft.com/fr/solve-problem/y%20%3D%20%20x%20%60cdot%20%20%20%60frac%7B%20%20%60left(%20x%2B1%20%20%60right)%20%20%20%20%7D%7B%202%20%20%7D
        // sideSize = (Sqrt(8 × compactSize + 1) - 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) - 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a Symmetric Matrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new SymmetricMatrix(diagonalSize);
        for (let col = 0, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row;
        }
        return matrix;
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield [row, column, value]
       *
       * @returns {Generator<[number, number, number], void, void>}
       */
      *upperRightEntries() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield [row, col, value];

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield value
       *
       * @returns {Generator<[number, number, number], void, void>}
       */
      *upperRightValues() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield value;

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }
    }
    SymmetricMatrix.prototype.klassType = 'SymmetricMatrix';
    class DistanceMatrix extends SymmetricMatrix {
      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isDistanceMatrix(value) {
        return SymmetricMatrix.isSymmetricMatrix(value) && value.klassSubType === 'DistanceMatrix';
      }
      constructor(sideSize) {
        super(sideSize);
        if (!this.isDistance()) {
          throw new TypeError('Provided arguments do no produce a distance matrix');
        }
      }
      set(rowIndex, columnIndex, value) {
        // distance matrix diagonal is 0
        if (rowIndex === columnIndex) value = 0;
        return super.set(rowIndex, columnIndex, value);
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }

        // ensure distance
        array = array.slice();
        array[index] = 0;
        return super.addCross(index, array);
      }
      toSymmetricMatrix() {
        return new SymmetricMatrix(this);
      }
      clone() {
        const matrix = new DistanceMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          if (row === col) continue;
          matrix.set(row, col, value);
        }
        return matrix;
      }

      /**
       * Compact format upper-right corner of matrix
       * no diagonal (only zeros)
       * iterable from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 0 1 2 3
       * B 1 0 4 5
       * C 2 4 0 6
       * D 3 5 6 0
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6]`
       *
       * length is S(i=0, n=sideSize-1) => 6 for a 4 side sized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;
        const compactLength = (diagonalSize - 1) * diagonalSize / 2;

        /** @type {number[]} */
        const compact = new Array(compactLength);
        for (let col = 1, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        if (compactSize === 0) {
          return new this(0);
        }

        // compactSize in Natural integer range ]0;∞]
        // compactSize = (sideSize * (sideSize - 1)) / 2
        // sideSize = (Sqrt(8 × compactSize + 1) + 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) + 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a DistanceMatrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new this(diagonalSize);
        for (let col = 1, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return matrix;
      }
    }
    DistanceMatrix.prototype.klassSubType = 'DistanceMatrix';
    class BaseView extends AbstractMatrix {
      constructor(matrix, rows, columns) {
        super();
        this.matrix = matrix;
        this.rows = rows;
        this.columns = columns;
      }
    }
    class MatrixColumnView extends BaseView {
      constructor(matrix, column) {
        checkColumnIndex(matrix, column);
        super(matrix, matrix.rows, 1);
        this.column = column;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.column, value);
        return this;
      }
      get(rowIndex) {
        return this.matrix.get(rowIndex, this.column);
      }
    }
    class MatrixColumnSelectionView extends BaseView {
      constructor(matrix, columnIndices) {
        checkColumnIndices(matrix, columnIndices);
        super(matrix, matrix.rows, columnIndices.length);
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
      }
    }
    class MatrixFlipColumnView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
      }
    }
    class MatrixFlipRowView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
      }
    }
    class MatrixRowView extends BaseView {
      constructor(matrix, row) {
        checkRowIndex(matrix, row);
        super(matrix, 1, matrix.columns);
        this.row = row;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.row, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.row, columnIndex);
      }
    }
    class MatrixRowSelectionView extends BaseView {
      constructor(matrix, rowIndices) {
        checkRowIndices(matrix, rowIndices);
        super(matrix, rowIndices.length, matrix.columns);
        this.rowIndices = rowIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
      }
    }
    class MatrixSelectionView extends BaseView {
      constructor(matrix, rowIndices, columnIndices) {
        checkRowIndices(matrix, rowIndices);
        checkColumnIndices(matrix, columnIndices);
        super(matrix, rowIndices.length, columnIndices.length);
        this.rowIndices = rowIndices;
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
      }
    }
    class MatrixSubView extends BaseView {
      constructor(matrix, startRow, endRow, startColumn, endColumn) {
        checkRange(matrix, startRow, endRow, startColumn, endColumn);
        super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
        this.startRow = startRow;
        this.startColumn = startColumn;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
      }
    }
    class MatrixTransposeView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.columns, matrix.rows);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(columnIndex, rowIndex);
      }
    }
    class WrapperMatrix1D extends AbstractMatrix {
      constructor(data, options = {}) {
        const {
          rows = 1
        } = options;
        if (data.length % rows !== 0) {
          throw new Error('the data length is not divisible by the number of rows');
        }
        super();
        this.rows = rows;
        this.columns = data.length / rows;
        this.data = data;
      }
      set(rowIndex, columnIndex, value) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        this.data[index] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        return this.data[index];
      }
      _calculateIndex(row, column) {
        return row * this.columns + column;
      }
    }
    class WrapperMatrix2D extends AbstractMatrix {
      constructor(data) {
        super();
        this.data = data;
        this.rows = data.length;
        this.columns = data[0].length;
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
    }
    function wrap(array, options) {
      if (isAnyArray.isAnyArray(array)) {
        if (array[0] && isAnyArray.isAnyArray(array[0])) {
          return new WrapperMatrix2D(array);
        } else {
          return new WrapperMatrix1D(array, options);
        }
      } else {
        throw new Error('the argument is not an array');
      }
    }
    class LuDecomposition {
      constructor(matrix) {
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        let lu = matrix.clone();
        let rows = lu.rows;
        let columns = lu.columns;
        let pivotVector = new Float64Array(rows);
        let pivotSign = 1;
        let i, j, k, p, s, t, v;
        let LUcolj, kmax;
        for (i = 0; i < rows; i++) {
          pivotVector[i] = i;
        }
        LUcolj = new Float64Array(rows);
        for (j = 0; j < columns; j++) {
          for (i = 0; i < rows; i++) {
            LUcolj[i] = lu.get(i, j);
          }
          for (i = 0; i < rows; i++) {
            kmax = Math.min(i, j);
            s = 0;
            for (k = 0; k < kmax; k++) {
              s += lu.get(i, k) * LUcolj[k];
            }
            LUcolj[i] -= s;
            lu.set(i, j, LUcolj[i]);
          }
          p = j;
          for (i = j + 1; i < rows; i++) {
            if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
              p = i;
            }
          }
          if (p !== j) {
            for (k = 0; k < columns; k++) {
              t = lu.get(p, k);
              lu.set(p, k, lu.get(j, k));
              lu.set(j, k, t);
            }
            v = pivotVector[p];
            pivotVector[p] = pivotVector[j];
            pivotVector[j] = v;
            pivotSign = -pivotSign;
          }
          if (j < rows && lu.get(j, j) !== 0) {
            for (i = j + 1; i < rows; i++) {
              lu.set(i, j, lu.get(i, j) / lu.get(j, j));
            }
          }
        }
        this.LU = lu;
        this.pivotVector = pivotVector;
        this.pivotSign = pivotSign;
      }
      isSingular() {
        let data = this.LU;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          if (data.get(j, j) === 0) {
            return true;
          }
        }
        return false;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let lu = this.LU;
        let rows = lu.rows;
        if (rows !== value.rows) {
          throw new Error('Invalid matrix dimensions');
        }
        if (this.isSingular()) {
          throw new Error('LU matrix is singular');
        }
        let count = value.columns;
        let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
        let columns = lu.columns;
        let i, j, k;
        for (k = 0; k < columns; k++) {
          for (i = k + 1; i < columns; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        for (k = columns - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / lu.get(k, k));
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        return X;
      }
      get determinant() {
        let data = this.LU;
        if (!data.isSquare()) {
          throw new Error('Matrix must be square');
        }
        let determinant = this.pivotSign;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          determinant *= data.get(j, j);
        }
        return determinant;
      }
      get lowerTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i > j) {
              X.set(i, j, data.get(i, j));
            } else if (i === j) {
              X.set(i, j, 1);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get upperTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i <= j) {
              X.set(i, j, data.get(i, j));
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get pivotPermutationVector() {
        return Array.from(this.pivotVector);
      }
    }
    function hypotenuse(a, b) {
      let r = 0;
      if (Math.abs(a) > Math.abs(b)) {
        r = b / a;
        return Math.abs(a) * Math.sqrt(1 + r * r);
      }
      if (b !== 0) {
        r = a / b;
        return Math.abs(b) * Math.sqrt(1 + r * r);
      }
      return 0;
    }
    class QrDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let qr = value.clone();
        let m = value.rows;
        let n = value.columns;
        let rdiag = new Float64Array(n);
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          let nrm = 0;
          for (i = k; i < m; i++) {
            nrm = hypotenuse(nrm, qr.get(i, k));
          }
          if (nrm !== 0) {
            if (qr.get(k, k) < 0) {
              nrm = -nrm;
            }
            for (i = k; i < m; i++) {
              qr.set(i, k, qr.get(i, k) / nrm);
            }
            qr.set(k, k, qr.get(k, k) + 1);
            for (j = k + 1; j < n; j++) {
              s = 0;
              for (i = k; i < m; i++) {
                s += qr.get(i, k) * qr.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < m; i++) {
                qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
              }
            }
          }
          rdiag[k] = -nrm;
        }
        this.QR = qr;
        this.Rdiag = rdiag;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let qr = this.QR;
        let m = qr.rows;
        if (value.rows !== m) {
          throw new Error('Matrix row dimensions must agree');
        }
        if (!this.isFullRank()) {
          throw new Error('Matrix is rank deficient');
        }
        let count = value.columns;
        let X = value.clone();
        let n = qr.columns;
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          for (j = 0; j < count; j++) {
            s = 0;
            for (i = k; i < m; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }
            s = -s / qr.get(k, k);
            for (i = k; i < m; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
        for (k = n - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / this.Rdiag[k]);
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
            }
          }
        }
        return X.subMatrix(0, n - 1, 0, count - 1);
      }
      isFullRank() {
        let columns = this.QR.columns;
        for (let i = 0; i < columns; i++) {
          if (this.Rdiag[i] === 0) {
            return false;
          }
        }
        return true;
      }
      get upperTriangularMatrix() {
        let qr = this.QR;
        let n = qr.columns;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            if (i < j) {
              X.set(i, j, qr.get(i, j));
            } else if (i === j) {
              X.set(i, j, this.Rdiag[i]);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get orthogonalMatrix() {
        let qr = this.QR;
        let rows = qr.rows;
        let columns = qr.columns;
        let X = new Matrix$1(rows, columns);
        let i, j, k, s;
        for (k = columns - 1; k >= 0; k--) {
          for (i = 0; i < rows; i++) {
            X.set(i, k, 0);
          }
          X.set(k, k, 1);
          for (j = k; j < columns; j++) {
            if (qr.get(k, k) !== 0) {
              s = 0;
              for (i = k; i < rows; i++) {
                s += qr.get(i, k) * X.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < rows; i++) {
                X.set(i, j, X.get(i, j) + s * qr.get(i, k));
              }
            }
          }
        }
        return X;
      }
    }
    class SingularValueDecomposition {
      constructor(value, options = {}) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (value.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let m = value.rows;
        let n = value.columns;
        const {
          computeLeftSingularVectors = true,
          computeRightSingularVectors = true,
          autoTranspose = false
        } = options;
        let wantu = Boolean(computeLeftSingularVectors);
        let wantv = Boolean(computeRightSingularVectors);
        let swapped = false;
        let a;
        if (m < n) {
          if (!autoTranspose) {
            a = value.clone();
            // eslint-disable-next-line no-console
            console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
          } else {
            a = value.transpose();
            m = a.rows;
            n = a.columns;
            swapped = true;
            let aux = wantu;
            wantu = wantv;
            wantv = aux;
          }
        } else {
          a = value.clone();
        }
        let nu = Math.min(m, n);
        let ni = Math.min(m + 1, n);
        let s = new Float64Array(ni);
        let U = new Matrix$1(m, nu);
        let V = new Matrix$1(n, n);
        let e = new Float64Array(n);
        let work = new Float64Array(m);
        let si = new Float64Array(ni);
        for (let i = 0; i < ni; i++) si[i] = i;
        let nct = Math.min(m - 1, n);
        let nrt = Math.max(0, Math.min(n - 2, m));
        let mrc = Math.max(nct, nrt);
        for (let k = 0; k < mrc; k++) {
          if (k < nct) {
            s[k] = 0;
            for (let i = k; i < m; i++) {
              s[k] = hypotenuse(s[k], a.get(i, k));
            }
            if (s[k] !== 0) {
              if (a.get(k, k) < 0) {
                s[k] = -s[k];
              }
              for (let i = k; i < m; i++) {
                a.set(i, k, a.get(i, k) / s[k]);
              }
              a.set(k, k, a.get(k, k) + 1);
            }
            s[k] = -s[k];
          }
          for (let j = k + 1; j < n; j++) {
            if (k < nct && s[k] !== 0) {
              let t = 0;
              for (let i = k; i < m; i++) {
                t += a.get(i, k) * a.get(i, j);
              }
              t = -t / a.get(k, k);
              for (let i = k; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * a.get(i, k));
              }
            }
            e[j] = a.get(k, j);
          }
          if (wantu && k < nct) {
            for (let i = k; i < m; i++) {
              U.set(i, k, a.get(i, k));
            }
          }
          if (k < nrt) {
            e[k] = 0;
            for (let i = k + 1; i < n; i++) {
              e[k] = hypotenuse(e[k], e[i]);
            }
            if (e[k] !== 0) {
              if (e[k + 1] < 0) {
                e[k] = 0 - e[k];
              }
              for (let i = k + 1; i < n; i++) {
                e[i] /= e[k];
              }
              e[k + 1] += 1;
            }
            e[k] = -e[k];
            if (k + 1 < m && e[k] !== 0) {
              for (let i = k + 1; i < m; i++) {
                work[i] = 0;
              }
              for (let i = k + 1; i < m; i++) {
                for (let j = k + 1; j < n; j++) {
                  work[i] += e[j] * a.get(i, j);
                }
              }
              for (let j = k + 1; j < n; j++) {
                let t = -e[j] / e[k + 1];
                for (let i = k + 1; i < m; i++) {
                  a.set(i, j, a.get(i, j) + t * work[i]);
                }
              }
            }
            if (wantv) {
              for (let i = k + 1; i < n; i++) {
                V.set(i, k, e[i]);
              }
            }
          }
        }
        let p = Math.min(n, m + 1);
        if (nct < n) {
          s[nct] = a.get(nct, nct);
        }
        if (m < p) {
          s[p - 1] = 0;
        }
        if (nrt + 1 < p) {
          e[nrt] = a.get(nrt, p - 1);
        }
        e[p - 1] = 0;
        if (wantu) {
          for (let j = nct; j < nu; j++) {
            for (let i = 0; i < m; i++) {
              U.set(i, j, 0);
            }
            U.set(j, j, 1);
          }
          for (let k = nct - 1; k >= 0; k--) {
            if (s[k] !== 0) {
              for (let j = k + 1; j < nu; j++) {
                let t = 0;
                for (let i = k; i < m; i++) {
                  t += U.get(i, k) * U.get(i, j);
                }
                t = -t / U.get(k, k);
                for (let i = k; i < m; i++) {
                  U.set(i, j, U.get(i, j) + t * U.get(i, k));
                }
              }
              for (let i = k; i < m; i++) {
                U.set(i, k, -U.get(i, k));
              }
              U.set(k, k, 1 + U.get(k, k));
              for (let i = 0; i < k - 1; i++) {
                U.set(i, k, 0);
              }
            } else {
              for (let i = 0; i < m; i++) {
                U.set(i, k, 0);
              }
              U.set(k, k, 1);
            }
          }
        }
        if (wantv) {
          for (let k = n - 1; k >= 0; k--) {
            if (k < nrt && e[k] !== 0) {
              for (let j = k + 1; j < n; j++) {
                let t = 0;
                for (let i = k + 1; i < n; i++) {
                  t += V.get(i, k) * V.get(i, j);
                }
                t = -t / V.get(k + 1, k);
                for (let i = k + 1; i < n; i++) {
                  V.set(i, j, V.get(i, j) + t * V.get(i, k));
                }
              }
            }
            for (let i = 0; i < n; i++) {
              V.set(i, k, 0);
            }
            V.set(k, k, 1);
          }
        }
        let pp = p - 1;
        let eps = Number.EPSILON;
        while (p > 0) {
          let k, kase;
          for (k = p - 2; k >= -1; k--) {
            if (k === -1) {
              break;
            }
            const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
            if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
              e[k] = 0;
              break;
            }
          }
          if (k === p - 2) {
            kase = 4;
          } else {
            let ks;
            for (ks = p - 1; ks >= k; ks--) {
              if (ks === k) {
                break;
              }
              let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
              if (Math.abs(s[ks]) <= eps * t) {
                s[ks] = 0;
                break;
              }
            }
            if (ks === k) {
              kase = 3;
            } else if (ks === p - 1) {
              kase = 1;
            } else {
              kase = 2;
              k = ks;
            }
          }
          k++;
          switch (kase) {
            case 1:
              {
                let f = e[p - 2];
                e[p - 2] = 0;
                for (let j = p - 2; j >= k; j--) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  if (j !== k) {
                    f = -sn * e[j - 1];
                    e[j - 1] = cs * e[j - 1];
                  }
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                      V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                      V.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 2:
              {
                let f = e[k - 1];
                e[k - 1] = 0;
                for (let j = k; j < p; j++) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  f = -sn * e[j];
                  e[j] = cs * e[j];
                  if (wantu) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                      U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                      U.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 3:
              {
                const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
                const sp = s[p - 1] / scale;
                const spm1 = s[p - 2] / scale;
                const epm1 = e[p - 2] / scale;
                const sk = s[k] / scale;
                const ek = e[k] / scale;
                const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
                const c = sp * epm1 * (sp * epm1);
                let shift = 0;
                if (b !== 0 || c !== 0) {
                  if (b < 0) {
                    shift = 0 - Math.sqrt(b * b + c);
                  } else {
                    shift = Math.sqrt(b * b + c);
                  }
                  shift = c / (b + shift);
                }
                let f = (sk + sp) * (sk - sp) + shift;
                let g = sk * ek;
                for (let j = k; j < p - 1; j++) {
                  let t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  let cs = f / t;
                  let sn = g / t;
                  if (j !== k) {
                    e[j - 1] = t;
                  }
                  f = cs * s[j] + sn * e[j];
                  e[j] = cs * e[j] - sn * s[j];
                  g = sn * s[j + 1];
                  s[j + 1] = cs * s[j + 1];
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                      V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                      V.set(i, j, t);
                    }
                  }
                  t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  cs = f / t;
                  sn = g / t;
                  s[j] = t;
                  f = cs * e[j] + sn * s[j + 1];
                  s[j + 1] = -sn * e[j] + cs * s[j + 1];
                  g = sn * e[j + 1];
                  e[j + 1] = cs * e[j + 1];
                  if (wantu && j < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                      U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                      U.set(i, j, t);
                    }
                  }
                }
                e[p - 2] = f;
                break;
              }
            case 4:
              {
                if (s[k] <= 0) {
                  s[k] = s[k] < 0 ? -s[k] : 0;
                  if (wantv) {
                    for (let i = 0; i <= pp; i++) {
                      V.set(i, k, -V.get(i, k));
                    }
                  }
                }
                while (k < pp) {
                  if (s[k] >= s[k + 1]) {
                    break;
                  }
                  let t = s[k];
                  s[k] = s[k + 1];
                  s[k + 1] = t;
                  if (wantv && k < n - 1) {
                    for (let i = 0; i < n; i++) {
                      t = V.get(i, k + 1);
                      V.set(i, k + 1, V.get(i, k));
                      V.set(i, k, t);
                    }
                  }
                  if (wantu && k < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = U.get(i, k + 1);
                      U.set(i, k + 1, U.get(i, k));
                      U.set(i, k, t);
                    }
                  }
                  k++;
                }
                p--;
                break;
              }
            // no default
          }
        }
        if (swapped) {
          let tmp = V;
          V = U;
          U = tmp;
        }
        this.m = m;
        this.n = n;
        this.s = s;
        this.U = U;
        this.V = V;
      }
      solve(value) {
        let Y = value;
        let e = this.threshold;
        let scols = this.s.length;
        let Ls = Matrix$1.zeros(scols, scols);
        for (let i = 0; i < scols; i++) {
          if (Math.abs(this.s[i]) <= e) {
            Ls.set(i, i, 0);
          } else {
            Ls.set(i, i, 1 / this.s[i]);
          }
        }
        let U = this.U;
        let V = this.rightSingularVectors;
        let VL = V.mmul(Ls);
        let vrows = V.rows;
        let urows = U.rows;
        let VLU = Matrix$1.zeros(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < scols; k++) {
              sum += VL.get(i, k) * U.get(j, k);
            }
            VLU.set(i, j, sum);
          }
        }
        return VLU.mmul(Y);
      }
      solveForDiagonal(value) {
        return this.solve(Matrix$1.diag(value));
      }
      inverse() {
        let V = this.V;
        let e = this.threshold;
        let vrows = V.rows;
        let vcols = V.columns;
        let X = new Matrix$1(vrows, this.s.length);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < vcols; j++) {
            if (Math.abs(this.s[j]) > e) {
              X.set(i, j, V.get(i, j) / this.s[j]);
            }
          }
        }
        let U = this.U;
        let urows = U.rows;
        let ucols = U.columns;
        let Y = new Matrix$1(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < ucols; k++) {
              sum += X.get(i, k) * U.get(j, k);
            }
            Y.set(i, j, sum);
          }
        }
        return Y;
      }
      get condition() {
        return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
      }
      get norm2() {
        return this.s[0];
      }
      get rank() {
        let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
        let r = 0;
        let s = this.s;
        for (let i = 0, ii = s.length; i < ii; i++) {
          if (s[i] > tol) {
            r++;
          }
        }
        return r;
      }
      get diagonal() {
        return Array.from(this.s);
      }
      get threshold() {
        return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
      }
      get leftSingularVectors() {
        return this.U;
      }
      get rightSingularVectors() {
        return this.V;
      }
      get diagonalMatrix() {
        return Matrix$1.diag(this.s);
      }
    }
    function inverse(matrix, useSVD = false) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      if (useSVD) {
        return new SingularValueDecomposition(matrix).inverse();
      } else {
        return solve(matrix, Matrix$1.eye(matrix.rows));
      }
    }
    function solve(leftHandSide, rightHandSide, useSVD = false) {
      leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
      rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
      if (useSVD) {
        return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
      } else {
        return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
      }
    }
    function determinant(matrix) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isSquare()) {
        if (matrix.columns === 0) {
          return 1;
        }
        let a, b, c, d;
        if (matrix.columns === 2) {
          // 2 x 2 matrix
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(1, 0);
          d = matrix.get(1, 1);
          return a * d - b * c;
        } else if (matrix.columns === 3) {
          // 3 x 3 matrix
          let subMatrix0, subMatrix1, subMatrix2;
          subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
          subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
          subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(0, 2);
          return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
        } else {
          // general purpose determinant using the LU decomposition
          return new LuDecomposition(matrix).determinant;
        }
      } else {
        throw Error('determinant can only be calculated for a square matrix');
      }
    }
    function xrange(n, exception) {
      let range = [];
      for (let i = 0; i < n; i++) {
        if (i !== exception) {
          range.push(i);
        }
      }
      return range;
    }
    function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
      if (error > thresholdError) {
        return new Array(matrix.rows + 1).fill(0);
      } else {
        let returnArray = matrix.addRow(index, [0]);
        for (let i = 0; i < returnArray.rows; i++) {
          if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
            returnArray.set(i, 0, 0);
          }
        }
        return returnArray.to1DArray();
      }
    }
    function linearDependencies(matrix, options = {}) {
      const {
        thresholdValue = 10e-10,
        thresholdError = 10e-10
      } = options;
      matrix = Matrix$1.checkMatrix(matrix);
      let n = matrix.rows;
      let results = new Matrix$1(n, n);
      for (let i = 0; i < n; i++) {
        let b = Matrix$1.columnVector(matrix.getRow(i));
        let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
        let svd = new SingularValueDecomposition(Abis);
        let x = svd.solve(b);
        let error = Matrix$1.sub(b, Abis.mmul(x)).abs().max();
        results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
      }
      return results;
    }
    function pseudoInverse(matrix, threshold = Number.EPSILON) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isEmpty()) {
        // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
        // (0xn)*(nx0)*(0xn) = 0xn
        // (nx0)*(0xn)*(nx0) = nx0
        return matrix.transpose();
      }
      let svdSolution = new SingularValueDecomposition(matrix, {
        autoTranspose: true
      });
      let U = svdSolution.leftSingularVectors;
      let V = svdSolution.rightSingularVectors;
      let s = svdSolution.diagonal;
      for (let i = 0; i < s.length; i++) {
        if (Math.abs(s[i]) > threshold) {
          s[i] = 1.0 / s[i];
        } else {
          s[i] = 0.0;
        }
      }
      return V.mmul(Matrix$1.diag(s).mmul(U.transpose()));
    }
    function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true
      } = options;
      if (center) {
        xMatrix = xMatrix.center('column');
        if (!yIsSame) {
          yMatrix = yMatrix.center('column');
        }
      }
      const cov = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < cov.rows; i++) {
        for (let j = 0; j < cov.columns; j++) {
          cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
        }
      }
      return cov;
    }
    function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true,
        scale = true
      } = options;
      if (center) {
        xMatrix.center('column');
        if (!yIsSame) {
          yMatrix.center('column');
        }
      }
      if (scale) {
        xMatrix.scale('column');
        if (!yIsSame) {
          yMatrix.scale('column');
        }
      }
      const sdx = xMatrix.standardDeviation('column', {
        unbiased: true
      });
      const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
        unbiased: true
      });
      const corr = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < corr.rows; i++) {
        for (let j = 0; j < corr.columns; j++) {
          corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
        }
      }
      return corr;
    }
    class EigenvalueDecomposition {
      constructor(matrix, options = {}) {
        const {
          assumeSymmetric = false
        } = options;
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        if (!matrix.isSquare()) {
          throw new Error('Matrix is not a square matrix');
        }
        if (matrix.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let n = matrix.columns;
        let V = new Matrix$1(n, n);
        let d = new Float64Array(n);
        let e = new Float64Array(n);
        let value = matrix;
        let i, j;
        let isSymmetric = false;
        if (assumeSymmetric) {
          isSymmetric = true;
        } else {
          isSymmetric = matrix.isSymmetric();
        }
        if (isSymmetric) {
          for (i = 0; i < n; i++) {
            for (j = 0; j < n; j++) {
              V.set(i, j, value.get(i, j));
            }
          }
          tred2(n, e, d, V);
          tql2(n, e, d, V);
        } else {
          let H = new Matrix$1(n, n);
          let ort = new Float64Array(n);
          for (j = 0; j < n; j++) {
            for (i = 0; i < n; i++) {
              H.set(i, j, value.get(i, j));
            }
          }
          orthes(n, H, ort, V);
          hqr2(n, e, d, V, H);
        }
        this.n = n;
        this.e = e;
        this.d = d;
        this.V = V;
      }
      get realEigenvalues() {
        return Array.from(this.d);
      }
      get imaginaryEigenvalues() {
        return Array.from(this.e);
      }
      get eigenvectorMatrix() {
        return this.V;
      }
      get diagonalMatrix() {
        let n = this.n;
        let e = this.e;
        let d = this.d;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            X.set(i, j, 0);
          }
          X.set(i, i, d[i]);
          if (e[i] > 0) {
            X.set(i, i + 1, e[i]);
          } else if (e[i] < 0) {
            X.set(i, i - 1, e[i]);
          }
        }
        return X;
      }
    }
    function tred2(n, e, d, V) {
      let f, g, h, i, j, k, hh, scale;
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
      }
      for (i = n - 1; i > 0; i--) {
        scale = 0;
        h = 0;
        for (k = 0; k < i; k++) {
          scale = scale + Math.abs(d[k]);
        }
        if (scale === 0) {
          e[i] = d[i - 1];
          for (j = 0; j < i; j++) {
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
            V.set(j, i, 0);
          }
        } else {
          for (k = 0; k < i; k++) {
            d[k] /= scale;
            h += d[k] * d[k];
          }
          f = d[i - 1];
          g = Math.sqrt(h);
          if (f > 0) {
            g = -g;
          }
          e[i] = scale * g;
          h = h - f * g;
          d[i - 1] = f - g;
          for (j = 0; j < i; j++) {
            e[j] = 0;
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            V.set(j, i, f);
            g = e[j] + V.get(j, j) * f;
            for (k = j + 1; k <= i - 1; k++) {
              g += V.get(k, j) * d[k];
              e[k] += V.get(k, j) * f;
            }
            e[j] = g;
          }
          f = 0;
          for (j = 0; j < i; j++) {
            e[j] /= h;
            f += e[j] * d[j];
          }
          hh = f / (h + h);
          for (j = 0; j < i; j++) {
            e[j] -= hh * d[j];
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            g = e[j];
            for (k = j; k <= i - 1; k++) {
              V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
            }
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
          }
        }
        d[i] = h;
      }
      for (i = 0; i < n - 1; i++) {
        V.set(n - 1, i, V.get(i, i));
        V.set(i, i, 1);
        h = d[i + 1];
        if (h !== 0) {
          for (k = 0; k <= i; k++) {
            d[k] = V.get(k, i + 1) / h;
          }
          for (j = 0; j <= i; j++) {
            g = 0;
            for (k = 0; k <= i; k++) {
              g += V.get(k, i + 1) * V.get(k, j);
            }
            for (k = 0; k <= i; k++) {
              V.set(k, j, V.get(k, j) - g * d[k]);
            }
          }
        }
        for (k = 0; k <= i; k++) {
          V.set(k, i + 1, 0);
        }
      }
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
        V.set(n - 1, j, 0);
      }
      V.set(n - 1, n - 1, 1);
      e[0] = 0;
    }
    function tql2(n, e, d, V) {
      let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
      for (i = 1; i < n; i++) {
        e[i - 1] = e[i];
      }
      e[n - 1] = 0;
      let f = 0;
      let tst1 = 0;
      let eps = Number.EPSILON;
      for (l = 0; l < n; l++) {
        tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
        m = l;
        while (m < n) {
          if (Math.abs(e[m]) <= eps * tst1) {
            break;
          }
          m++;
        }
        if (m > l) {
          do {
            g = d[l];
            p = (d[l + 1] - g) / (2 * e[l]);
            r = hypotenuse(p, 1);
            if (p < 0) {
              r = -r;
            }
            d[l] = e[l] / (p + r);
            d[l + 1] = e[l] * (p + r);
            dl1 = d[l + 1];
            h = g - d[l];
            for (i = l + 2; i < n; i++) {
              d[i] -= h;
            }
            f = f + h;
            p = d[m];
            c = 1;
            c2 = c;
            c3 = c;
            el1 = e[l + 1];
            s = 0;
            s2 = 0;
            for (i = m - 1; i >= l; i--) {
              c3 = c2;
              c2 = c;
              s2 = s;
              g = c * e[i];
              h = c * p;
              r = hypotenuse(p, e[i]);
              e[i + 1] = s * r;
              s = e[i] / r;
              c = p / r;
              p = c * d[i] - s * g;
              d[i + 1] = h + s * (c * g + s * d[i]);
              for (k = 0; k < n; k++) {
                h = V.get(k, i + 1);
                V.set(k, i + 1, s * V.get(k, i) + c * h);
                V.set(k, i, c * V.get(k, i) - s * h);
              }
            }
            p = -s * s2 * c3 * el1 * e[l] / dl1;
            e[l] = s * p;
            d[l] = c * p;
          } while (Math.abs(e[l]) > eps * tst1);
        }
        d[l] = d[l] + f;
        e[l] = 0;
      }
      for (i = 0; i < n - 1; i++) {
        k = i;
        p = d[i];
        for (j = i + 1; j < n; j++) {
          if (d[j] < p) {
            k = j;
            p = d[j];
          }
        }
        if (k !== i) {
          d[k] = d[i];
          d[i] = p;
          for (j = 0; j < n; j++) {
            p = V.get(j, i);
            V.set(j, i, V.get(j, k));
            V.set(j, k, p);
          }
        }
      }
    }
    function orthes(n, H, ort, V) {
      let low = 0;
      let high = n - 1;
      let f, g, h, i, j, m;
      let scale;
      for (m = low + 1; m <= high - 1; m++) {
        scale = 0;
        for (i = m; i <= high; i++) {
          scale = scale + Math.abs(H.get(i, m - 1));
        }
        if (scale !== 0) {
          h = 0;
          for (i = high; i >= m; i--) {
            ort[i] = H.get(i, m - 1) / scale;
            h += ort[i] * ort[i];
          }
          g = Math.sqrt(h);
          if (ort[m] > 0) {
            g = -g;
          }
          h = h - ort[m] * g;
          ort[m] = ort[m] - g;
          for (j = m; j < n; j++) {
            f = 0;
            for (i = high; i >= m; i--) {
              f += ort[i] * H.get(i, j);
            }
            f = f / h;
            for (i = m; i <= high; i++) {
              H.set(i, j, H.get(i, j) - f * ort[i]);
            }
          }
          for (i = 0; i <= high; i++) {
            f = 0;
            for (j = high; j >= m; j--) {
              f += ort[j] * H.get(i, j);
            }
            f = f / h;
            for (j = m; j <= high; j++) {
              H.set(i, j, H.get(i, j) - f * ort[j]);
            }
          }
          ort[m] = scale * ort[m];
          H.set(m, m - 1, scale * g);
        }
      }
      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          V.set(i, j, i === j ? 1 : 0);
        }
      }
      for (m = high - 1; m >= low + 1; m--) {
        if (H.get(m, m - 1) !== 0) {
          for (i = m + 1; i <= high; i++) {
            ort[i] = H.get(i, m - 1);
          }
          for (j = m; j <= high; j++) {
            g = 0;
            for (i = m; i <= high; i++) {
              g += ort[i] * V.get(i, j);
            }
            g = g / ort[m] / H.get(m, m - 1);
            for (i = m; i <= high; i++) {
              V.set(i, j, V.get(i, j) + g * ort[i]);
            }
          }
        }
      }
    }
    function hqr2(nn, e, d, V, H) {
      let n = nn - 1;
      let low = 0;
      let high = nn - 1;
      let eps = Number.EPSILON;
      let exshift = 0;
      let norm = 0;
      let p = 0;
      let q = 0;
      let r = 0;
      let s = 0;
      let z = 0;
      let iter = 0;
      let i, j, k, l, m, t, w, x, y;
      let ra, sa, vr, vi;
      let notlast, cdivres;
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          d[i] = H.get(i, i);
          e[i] = 0;
        }
        for (j = Math.max(i - 1, 0); j < nn; j++) {
          norm = norm + Math.abs(H.get(i, j));
        }
      }
      while (n >= low) {
        l = n;
        while (l > low) {
          s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
          if (s === 0) {
            s = norm;
          }
          if (Math.abs(H.get(l, l - 1)) < eps * s) {
            break;
          }
          l--;
        }
        if (l === n) {
          H.set(n, n, H.get(n, n) + exshift);
          d[n] = H.get(n, n);
          e[n] = 0;
          n--;
          iter = 0;
        } else if (l === n - 1) {
          w = H.get(n, n - 1) * H.get(n - 1, n);
          p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
          q = p * p + w;
          z = Math.sqrt(Math.abs(q));
          H.set(n, n, H.get(n, n) + exshift);
          H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
          x = H.get(n, n);
          if (q >= 0) {
            z = p >= 0 ? p + z : p - z;
            d[n - 1] = x + z;
            d[n] = d[n - 1];
            if (z !== 0) {
              d[n] = x - w / z;
            }
            e[n - 1] = 0;
            e[n] = 0;
            x = H.get(n, n - 1);
            s = Math.abs(x) + Math.abs(z);
            p = x / s;
            q = z / s;
            r = Math.sqrt(p * p + q * q);
            p = p / r;
            q = q / r;
            for (j = n - 1; j < nn; j++) {
              z = H.get(n - 1, j);
              H.set(n - 1, j, q * z + p * H.get(n, j));
              H.set(n, j, q * H.get(n, j) - p * z);
            }
            for (i = 0; i <= n; i++) {
              z = H.get(i, n - 1);
              H.set(i, n - 1, q * z + p * H.get(i, n));
              H.set(i, n, q * H.get(i, n) - p * z);
            }
            for (i = low; i <= high; i++) {
              z = V.get(i, n - 1);
              V.set(i, n - 1, q * z + p * V.get(i, n));
              V.set(i, n, q * V.get(i, n) - p * z);
            }
          } else {
            d[n - 1] = x + p;
            d[n] = x + p;
            e[n - 1] = z;
            e[n] = -z;
          }
          n = n - 2;
          iter = 0;
        } else {
          x = H.get(n, n);
          y = 0;
          w = 0;
          if (l < n) {
            y = H.get(n - 1, n - 1);
            w = H.get(n, n - 1) * H.get(n - 1, n);
          }
          if (iter === 10) {
            exshift += x;
            for (i = low; i <= n; i++) {
              H.set(i, i, H.get(i, i) - x);
            }
            s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
            // eslint-disable-next-line no-multi-assign
            x = y = 0.75 * s;
            w = -0.4375 * s * s;
          }
          if (iter === 30) {
            s = (y - x) / 2;
            s = s * s + w;
            if (s > 0) {
              s = Math.sqrt(s);
              if (y < x) {
                s = -s;
              }
              s = x - w / ((y - x) / 2 + s);
              for (i = low; i <= n; i++) {
                H.set(i, i, H.get(i, i) - s);
              }
              exshift += s;
              // eslint-disable-next-line no-multi-assign
              x = y = w = 0.964;
            }
          }
          iter = iter + 1;
          m = n - 2;
          while (m >= l) {
            z = H.get(m, m);
            r = x - z;
            s = y - z;
            p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
            q = H.get(m + 1, m + 1) - z - r - s;
            r = H.get(m + 2, m + 1);
            s = Math.abs(p) + Math.abs(q) + Math.abs(r);
            p = p / s;
            q = q / s;
            r = r / s;
            if (m === l) {
              break;
            }
            if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
              break;
            }
            m--;
          }
          for (i = m + 2; i <= n; i++) {
            H.set(i, i - 2, 0);
            if (i > m + 2) {
              H.set(i, i - 3, 0);
            }
          }
          for (k = m; k <= n - 1; k++) {
            notlast = k !== n - 1;
            if (k !== m) {
              p = H.get(k, k - 1);
              q = H.get(k + 1, k - 1);
              r = notlast ? H.get(k + 2, k - 1) : 0;
              x = Math.abs(p) + Math.abs(q) + Math.abs(r);
              if (x !== 0) {
                p = p / x;
                q = q / x;
                r = r / x;
              }
            }
            if (x === 0) {
              break;
            }
            s = Math.sqrt(p * p + q * q + r * r);
            if (p < 0) {
              s = -s;
            }
            if (s !== 0) {
              if (k !== m) {
                H.set(k, k - 1, -s * x);
              } else if (l !== m) {
                H.set(k, k - 1, -H.get(k, k - 1));
              }
              p = p + s;
              x = p / s;
              y = q / s;
              z = r / s;
              q = q / p;
              r = r / p;
              for (j = k; j < nn; j++) {
                p = H.get(k, j) + q * H.get(k + 1, j);
                if (notlast) {
                  p = p + r * H.get(k + 2, j);
                  H.set(k + 2, j, H.get(k + 2, j) - p * z);
                }
                H.set(k, j, H.get(k, j) - p * x);
                H.set(k + 1, j, H.get(k + 1, j) - p * y);
              }
              for (i = 0; i <= Math.min(n, k + 3); i++) {
                p = x * H.get(i, k) + y * H.get(i, k + 1);
                if (notlast) {
                  p = p + z * H.get(i, k + 2);
                  H.set(i, k + 2, H.get(i, k + 2) - p * r);
                }
                H.set(i, k, H.get(i, k) - p);
                H.set(i, k + 1, H.get(i, k + 1) - p * q);
              }
              for (i = low; i <= high; i++) {
                p = x * V.get(i, k) + y * V.get(i, k + 1);
                if (notlast) {
                  p = p + z * V.get(i, k + 2);
                  V.set(i, k + 2, V.get(i, k + 2) - p * r);
                }
                V.set(i, k, V.get(i, k) - p);
                V.set(i, k + 1, V.get(i, k + 1) - p * q);
              }
            }
          }
        }
      }
      if (norm === 0) {
        return;
      }
      for (n = nn - 1; n >= 0; n--) {
        p = d[n];
        q = e[n];
        if (q === 0) {
          l = n;
          H.set(n, n, 1);
          for (i = n - 1; i >= 0; i--) {
            w = H.get(i, i) - p;
            r = 0;
            for (j = l; j <= n; j++) {
              r = r + H.get(i, j) * H.get(j, n);
            }
            if (e[i] < 0) {
              z = w;
              s = r;
            } else {
              l = i;
              if (e[i] === 0) {
                H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
                t = (x * s - z * r) / q;
                H.set(i, n, t);
                H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
              }
              t = Math.abs(H.get(i, n));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        } else if (q < 0) {
          l = n - 1;
          if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
            H.set(n - 1, n - 1, q / H.get(n, n - 1));
            H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
          } else {
            cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
            H.set(n - 1, n - 1, cdivres[0]);
            H.set(n - 1, n, cdivres[1]);
          }
          H.set(n, n - 1, 0);
          H.set(n, n, 1);
          for (i = n - 2; i >= 0; i--) {
            ra = 0;
            sa = 0;
            for (j = l; j <= n; j++) {
              ra = ra + H.get(i, j) * H.get(j, n - 1);
              sa = sa + H.get(i, j) * H.get(j, n);
            }
            w = H.get(i, i) - p;
            if (e[i] < 0) {
              z = w;
              r = ra;
              s = sa;
            } else {
              l = i;
              if (e[i] === 0) {
                cdivres = cdiv(-ra, -sa, w, q);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
                vi = (d[i] - p) * 2 * q;
                if (vr === 0 && vi === 0) {
                  vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
                }
                cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
                if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
                  H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
                  H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
                } else {
                  cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
                  H.set(i + 1, n - 1, cdivres[0]);
                  H.set(i + 1, n, cdivres[1]);
                }
              }
              t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n - 1, H.get(j, n - 1) / t);
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        }
      }
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          for (j = i; j < nn; j++) {
            V.set(i, j, H.get(i, j));
          }
        }
      }
      for (j = nn - 1; j >= low; j--) {
        for (i = low; i <= high; i++) {
          z = 0;
          for (k = low; k <= Math.min(j, high); k++) {
            z = z + V.get(i, k) * H.get(k, j);
          }
          V.set(i, j, z);
        }
      }
    }
    function cdiv(xr, xi, yr, yi) {
      let r, d;
      if (Math.abs(yr) > Math.abs(yi)) {
        r = yi / yr;
        d = yr + r * yi;
        return [(xr + r * xi) / d, (xi - r * xr) / d];
      } else {
        r = yr / yi;
        d = yi + r * yr;
        return [(r * xr + xi) / d, (r * xi - xr) / d];
      }
    }
    class CholeskyDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (!value.isSymmetric()) {
          throw new Error('Matrix is not symmetric');
        }
        let a = value;
        let dimension = a.rows;
        let l = new Matrix$1(dimension, dimension);
        let positiveDefinite = true;
        let i, j, k;
        for (j = 0; j < dimension; j++) {
          let d = 0;
          for (k = 0; k < j; k++) {
            let s = 0;
            for (i = 0; i < k; i++) {
              s += l.get(k, i) * l.get(j, i);
            }
            s = (a.get(j, k) - s) / l.get(k, k);
            l.set(j, k, s);
            d = d + s * s;
          }
          d = a.get(j, j) - d;
          positiveDefinite &&= d > 0;
          l.set(j, j, Math.sqrt(Math.max(d, 0)));
          for (k = j + 1; k < dimension; k++) {
            l.set(j, k, 0);
          }
        }
        this.L = l;
        this.positiveDefinite = positiveDefinite;
      }
      isPositiveDefinite() {
        return this.positiveDefinite;
      }
      solve(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let l = this.L;
        let dimension = l.rows;
        if (value.rows !== dimension) {
          throw new Error('Matrix dimensions do not match');
        }
        if (this.isPositiveDefinite() === false) {
          throw new Error('Matrix is not positive definite');
        }
        let count = value.columns;
        let B = value.clone();
        let i, j, k;
        for (k = 0; k < dimension; k++) {
          for (j = 0; j < count; j++) {
            for (i = 0; i < k; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        for (k = dimension - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            for (i = k + 1; i < dimension; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        return B;
      }
      get lowerTriangularMatrix() {
        return this.L;
      }
    }
    class nipals {
      constructor(X, options = {}) {
        X = WrapperMatrix2D.checkMatrix(X);
        let {
          Y
        } = options;
        const {
          scaleScores = false,
          maxIterations = 1000,
          terminationCriteria = 1e-10
        } = options;
        let u;
        if (Y) {
          if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
            Y = Matrix$1.columnVector(Y);
          } else {
            Y = WrapperMatrix2D.checkMatrix(Y);
          }
          if (Y.rows !== X.rows) {
            throw new Error('Y should have the same number of rows as X');
          }
          u = Y.getColumnVector(0);
        } else {
          u = X.getColumnVector(0);
        }
        let diff = 1;
        let t, q, w, tOld;
        for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
          w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
          w = w.div(w.norm());
          t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
          if (counter > 0) {
            diff = t.clone().sub(tOld).pow(2).sum();
          }
          tOld = t.clone();
          if (Y) {
            q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
            q = q.div(q.norm());
            u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
          } else {
            u = t;
          }
        }
        if (Y) {
          let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          p = p.div(p.norm());
          let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
          let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
          this.t = t;
          this.p = p.transpose();
          this.w = w.transpose();
          this.q = q;
          this.u = u;
          this.s = t.transpose().mmul(t);
          this.xResidual = xResidual;
          this.yResidual = yResidual;
          this.betas = residual;
        } else {
          this.w = w.transpose();
          this.s = t.transpose().mmul(t).sqrt();
          if (scaleScores) {
            this.t = t.clone().div(this.s.get(0, 0));
          } else {
            this.t = t;
          }
          this.xResidual = X.sub(t.mmul(w.transpose()));
        }
      }
    }
    matrix.AbstractMatrix = AbstractMatrix;
    matrix.CHO = CholeskyDecomposition;
    matrix.CholeskyDecomposition = CholeskyDecomposition;
    matrix.DistanceMatrix = DistanceMatrix;
    matrix.EVD = EigenvalueDecomposition;
    matrix.EigenvalueDecomposition = EigenvalueDecomposition;
    matrix.LU = LuDecomposition;
    matrix.LuDecomposition = LuDecomposition;
    var Matrix_1 = matrix.Matrix = Matrix$1;
    matrix.MatrixColumnSelectionView = MatrixColumnSelectionView;
    matrix.MatrixColumnView = MatrixColumnView;
    matrix.MatrixFlipColumnView = MatrixFlipColumnView;
    matrix.MatrixFlipRowView = MatrixFlipRowView;
    matrix.MatrixRowSelectionView = MatrixRowSelectionView;
    matrix.MatrixRowView = MatrixRowView;
    matrix.MatrixSelectionView = MatrixSelectionView;
    matrix.MatrixSubView = MatrixSubView;
    matrix.MatrixTransposeView = MatrixTransposeView;
    matrix.NIPALS = nipals;
    matrix.Nipals = nipals;
    matrix.QR = QrDecomposition;
    matrix.QrDecomposition = QrDecomposition;
    matrix.SVD = SingularValueDecomposition;
    matrix.SingularValueDecomposition = SingularValueDecomposition;
    matrix.SymmetricMatrix = SymmetricMatrix;
    matrix.WrapperMatrix1D = WrapperMatrix1D;
    matrix.WrapperMatrix2D = WrapperMatrix2D;
    matrix.correlation = correlation;
    matrix.covariance = covariance;
    var _default = matrix.default = Matrix$1;
    matrix.determinant = determinant;
    matrix.inverse = inverse;
    matrix.linearDependencies = linearDependencies;
    matrix.pseudoInverse = pseudoInverse;
    matrix.solve = solve;
    matrix.wrap = wrap;

    const Matrix = Matrix_1;
    _default.Matrix ? _default.Matrix : Matrix_1;

    /**
     * Algorithm that finds the shortest distance from one node to the other
     * @param {Matrix} adjMatrix - A squared adjacency matrix
     * @return {Matrix} - Distance from a node to the other, -1 if the node is unreachable
     */
    function floydWarshall(adjMatrix) {
      if (Matrix.isMatrix(adjMatrix) && adjMatrix.columns !== adjMatrix.rows) {
        throw new TypeError('The adjacency matrix should be squared');
      }
      const numVertices = adjMatrix.columns;
      let distMatrix = new Matrix(numVertices, numVertices);
      distMatrix.apply((row, column) => {
        // principal diagonal is 0
        if (row === column) {
          distMatrix.set(row, column, 0);
        } else {
          let val = adjMatrix.get(row, column);
          if (val || Object.is(val, -0)) {
            // edges values remain the same
            distMatrix.set(row, column, val);
          } else {
            // 0 values become infinity
            distMatrix.set(row, column, Number.POSITIVE_INFINITY);
          }
        }
      });
      for (let k = 0; k < numVertices; ++k) {
        for (let i = 0; i < numVertices; ++i) {
          for (let j = 0; j < numVertices; ++j) {
            let dist = distMatrix.get(i, k) + distMatrix.get(k, j);
            if (distMatrix.get(i, j) > dist) {
              distMatrix.set(i, j, dist);
            }
          }
        }
      }
      // When there's no connection the value is -1
      distMatrix.apply((row, column) => {
        if (distMatrix.get(row, column) === Number.POSITIVE_INFINITY) {
          distMatrix.set(row, column, -1);
        }
      });
      return distMatrix;
    }

    /**
     * Returns a connectivity matrix
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {boolean} [options.pathLength=false] - get the path length between atoms
     * @param {boolean} [options.mass=false] - set the nominal mass of the atoms on diagonal
     * @param {boolean} [options.atomicNo=false] - set the atomic number of the atom on diagonal
     * @param {boolean} [options.negativeAtomicNo=false] - set the atomic number * -1 of the atom on diagonal
     * @param {boolean} [options.sdt=false] - set 1, 2 or 3 depending if single, double or triple bond
     * @param {boolean} [options.sdta=false] - set 1, 2, 3 or 4 depending if single, double, triple or aromatic  bond
     */
    function getConnectivityMatrix(molecule, options = {}) {
      const OCL = molecule.getOCL();
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      const nbAtoms = molecule.getAllAtoms();
      let result = new Array(nbAtoms).fill();
      result = result.map(() => new Array(nbAtoms).fill(0));
      if (!options.pathLength) {
        if (options.atomicNo) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = molecule.getAtomicNo(i);
          }
        } else if (options.negativeAtomicNo) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = -molecule.getAtomicNo(i);
          }
        } else if (options.mass) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = OCL.Molecule.cRoundedMass[molecule.getAtomicNo(i)];
          }
        } else {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = 1;
          }
        }
      }
      if (options.sdt) {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
          }
        }
      } else if (options.sdta) {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            const bondNumber = molecule.getConnBond(i, j);
            if (molecule.isAromaticBond(bondNumber)) {
              result[i][molecule.getConnAtom(i, j)] = 4;
            } else {
              result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
            }
          }
        }
      } else {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            result[i][molecule.getConnAtom(i, j)] = 1;
          }
        }
      }
      if (options.pathLength) {
        result = floydWarshall(new Matrix(result)).to2DArray();
      }
      return result;
    }

    /**
     * Creates a compact copy of the molecule without custom labels.
     * We don't want that custom labels interfere with hose code and diaID generation.
     * @param molecule
     * @returns
     */
    function getCompactCopyWithoutCustomLabels(molecule) {
      const tempMolecule = molecule.getCompactCopy();
      for (let i = 0; i < tempMolecule.getAllAtoms(); i++) {
        tempMolecule.setAtomCustomLabel(i, '');
      }
      return tempMolecule;
    }

    function getCanonizedDiaIDs(diaMol, options) {
      const {
        logger,
        maxNbAtoms
      } = options;
      const moleculeWithH = diaMol.moleculeWithH;
      if (moleculeWithH.getAllAtoms() > maxNbAtoms) {
        logger.warn(`too many atoms to evaluate heterotopic chiral bonds: ${moleculeWithH.getAllAtoms()} > ${maxNbAtoms}`);
        return [];
      }
      const heterotopicSymmetryRanks = diaMol.heterotopicSymmetryRanks;
      const finalRanks = diaMol.finalRanks;
      const canonizedDiaIDs = new Array(moleculeWithH.getAllAtoms());
      moleculeWithH.ensureHelperArrays(diaMol.molecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const cache = {};
      for (let i = 0; i < diaMol.moleculeWithH.getAllAtoms(); i++) {
        const rank = heterotopicSymmetryRanks[i];
        if (rank && cache[rank]) {
          canonizedDiaIDs[finalRanks[i]] = cache[rank].diaID;
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(diaMol.moleculeWithH);
        tagAtom(tempMolecule, i);
        makeRacemic(tempMolecule);
        const diaID = tempMolecule.getCanonizedIDCode(diaMol.molecule.getOCL().Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        canonizedDiaIDs[finalRanks[i]] = diaID;
      }
      return canonizedDiaIDs;
    }

    /**
     * Returns an array of strings (idCodes) specified molecule. Each string corresponds to a
     * hose code. By default it will calculate the hose codes for sphere 0 to 4 and will reuse
     * the existing tagged atoms.
     * @param molecule - The OCL molecule to process.
     * @param options - Options for generating hose codes.
     * @returns An array of hose code strings.
     */
    function getHoseCodesForAtomsAsStrings(molecule, options = {}) {
      const fragments = getHoseCodesForAtomsAsFragments(molecule, options);
      const OCL = molecule.getOCL();
      const hoses = [];
      for (const fragment of fragments) {
        hoses.push(fragment.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS));
      }
      return hoses;
    }

    /**
     * Get the canonized hose codes for a topic molecule. It will use the moleculeWithH
     * @param topicMolecule - The topic molecule to get the hose codes for.
     * @returns The canonized hose codes.
     */
    function getCanonizedHoseCodes(topicMolecule) {
      const options = topicMolecule.options;
      const heterotopicSymmetryRanks = topicMolecule.heterotopicSymmetryRanks;
      const moleculeWithH = topicMolecule.moleculeWithH;
      const finalRanks = topicMolecule.finalRanks;
      const canonizedHoseCodes = new Array(moleculeWithH.getAllAtoms());
      moleculeWithH.ensureHelperArrays(topicMolecule.molecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const cache = {};
      for (let i = 0; i < topicMolecule.moleculeWithH.getAllAtoms(); i++) {
        const rank = heterotopicSymmetryRanks[i];
        if (rank && cache[rank]) {
          canonizedHoseCodes[finalRanks[i]] = cache[rank].diaID;
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(moleculeWithH);
        tagAtom(tempMolecule, i);
        const hoses = getHoseCodesForAtomsAsStrings(tempMolecule, options);
        canonizedHoseCodes[finalRanks[i]] = hoses;
      }
      return canonizedHoseCodes;
    }

    /**
     * For each atom we will return an array of objects
     * @param topicMolecule
     * @param options
     * @returns
     */
    function getCanonizedHoseCodesForPath(topicMolecule, options = {}) {
      const {
        minPathLength = 0,
        maxPathLength = topicMolecule.options.maxPathLength,
        toAtomicNo,
        fromAtomicNo
      } = options;
      if (maxPathLength > topicMolecule.options.maxPathLength) {
        throw new Error(`maxPathLength cannot be larger than the one defined in topicMolecule: ${topicMolecule.options.maxPathLength}`);
      }
      const atomsPaths = topicMolecule.atomsPaths;
      const molecule = topicMolecule.moleculeWithH;
      const results = new Array(molecule.getAllAtoms());
      for (let fromAtom = 0; fromAtom < molecule.getAllAtoms(); fromAtom++) {
        results[fromAtom] = {
          fromDiaID: topicMolecule.diaIDs[fromAtom],
          paths: []
        };
        if (fromAtomicNo && molecule.getAtomicNo(fromAtom) !== fromAtomicNo) {
          continue;
        }
        for (let pathLength = minPathLength; pathLength <= maxPathLength; pathLength++) {
          const pathOfSpecificLength = atomsPaths[fromAtom][pathLength];
          for (const path of pathOfSpecificLength) {
            const toAtom = path.path.at(-1);
            if (toAtomicNo && molecule.getAtomicNo(toAtom) !== toAtomicNo) {
              continue;
            }
            results[fromAtom].paths.push({
              toDiaID: topicMolecule.diaIDs[toAtom],
              pathLength: path.pathLength,
              path: path.path,
              hoses: getHoseCodesForAtomsAsStrings(topicMolecule.moleculeWithH, {
                ...options,
                rootAtoms: path.path,
                tagAtoms: [fromAtom, toAtom]
              })
            });
          }
        }
      }
      return results;
    }

    function getDiaIDsAndInfo(diaMol, canonizedDiaIDs) {
      const newDiaIDs = [];
      const molecule = diaMol.moleculeWithH;
      const counts = {};
      for (const diaID of canonizedDiaIDs) {
        if (!counts[diaID]) {
          counts[diaID] = 0;
        }
        counts[diaID]++;
      }
      for (let i = 0; i < canonizedDiaIDs.length; i++) {
        const diaID = canonizedDiaIDs[diaMol.finalRanks[i]];
        if (!diaID) {
          throw new Error(`Unexpected missing canonized diaID for atom ${i}`);
        }
        const count = counts[diaID];
        if (!count) {
          throw new Error(`Unexpected missing count for diaID ${diaID}`);
        }
        const newDiaID = {
          idCode: diaID,
          attachedHydrogensIDCodes: [],
          attachedHydrogens: [],
          nbAttachedHydrogens: 0,
          atomLabel: molecule.getAtomLabel(i),
          nbEquivalentAtoms: count,
          heavyAtom: undefined,
          atomMapNo: molecule.getAtomMapNo(i)
        };
        if (molecule.getAtomicNo(i) === 1) {
          const atom = molecule.getConnAtom(i, 0);
          newDiaID.heavyAtom = canonizedDiaIDs[diaMol.finalRanks[atom]];
        }
        for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
          const atom = molecule.getConnAtom(i, j);
          if (molecule.getAtomicNo(atom) === 1) {
            newDiaID.nbAttachedHydrogens++;
            newDiaID.attachedHydrogens.push(atom);
            const hydrogenDiaID = canonizedDiaIDs[diaMol.finalRanks[atom]];
            if (!hydrogenDiaID) {
              throw new Error(`Unexpected missing canonized diaID for atom ${atom}`);
            }
            if (!newDiaID.attachedHydrogensIDCodes.includes(hydrogenDiaID)) {
              newDiaID.attachedHydrogensIDCodes.push(hydrogenDiaID);
            }
          }
        }
        newDiaIDs.push(newDiaID);
      }
      return newDiaIDs;
    }

    /**
     * Get a unique atomic number for a X
     * @param xMolecule
     * @returns
     */
    function getHeterotopicSymmetryRanks(xMolecule) {
      xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const symmetryRanks = [];
      for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
        symmetryRanks.push(xMolecule.getSymmetryRank(i));
      }
      return symmetryRanks;
    }
    function getFinalRanks(xMolecule) {
      xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      return xMolecule.getFinalRanks(0).map(rank => rank - 1);
    }

    /**
     * Returns the atoms that are chiral or pseudo chiral.
     * There could be some issues if the original molecule lacks chiral bonds.
     * The function will add them and this could lead to some issues in the case of pseudochiral atoms.
     * @param {import('openchemlib').Molecule} molecule
     * @returns {number[]}
     */
    function getChiralOrHeterotopicCarbons(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      const xAtomicNumber = getXAtomicNumber(molecule);
      const internalMolecule = molecule.getCompactCopy();
      // hydrogens may be diastereotopic, we need to add them
      internalMolecule.addImplicitHydrogens();
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (internalMolecule.getAtomicNo(i) === 1) {
          internalMolecule.setAtomicNo(i, xAtomicNumber);
        }
      }
      addPossibleChiralBonds(internalMolecule);
      internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const atoms = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (internalMolecule.getAtomicNo(i) === xAtomicNumber) {
          continue;
        }
        if (molecule.getAtomicNo(i) !== internalMolecule.getAtomicNo(i)) {
          throw new Error('getChiralOrHeterotopicCarbons: mismatching atomic numbers');
        }
        if (internalMolecule.getAtomicNo(i) !== 6) {
          continue;
        }
        const neighbourSymmetries = getNeighbourSymmetries(internalMolecule, i);
        if (neighbourSymmetries.length === 4) {
          atoms.push(i);
        }
      }
      return atoms;
    }
    function addPossibleChiralBonds(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      for (let i = 0; i < molecule.getAtoms(); i++) {
        if (molecule.getAtomicNo(i) !== 6) continue;
        if (molecule.getStereoBond(i) >= 0) continue;
        const neighbourSymmetries = getNeighbourSymmetries(molecule, i);
        if (neighbourSymmetries.length <= 2) continue;
        const stereoBond = molecule.getAtomPreferredStereoBond(i);
        if (stereoBond !== -1) {
          molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
          if (molecule.getBondAtom(1, stereoBond) === i) {
            const connAtom = molecule.getBondAtom(0, stereoBond);
            molecule.setBondAtom(0, stereoBond, i);
            molecule.setBondAtom(1, stereoBond, connAtom);
          }
          // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0);
        }
      }
    }
    function getNeighbourSymmetries(molecule, iAtom) {
      const neighbourSymmetries = [];
      for (let j = 0; j < molecule.getAllConnAtoms(iAtom); j++) {
        const connAtom = molecule.getConnAtom(iAtom, j);
        const symmetryRank = molecule.getSymmetryRank(connAtom);
        if (!neighbourSymmetries.includes(symmetryRank)) {
          neighbourSymmetries.push(molecule.getSymmetryRank(connAtom));
        }
      }
      return neighbourSymmetries;
    }

    /**
     * This function will add missing chiral bonds on carbons ensure that all enantiotopic
     * or diastereotopic atoms can be identified uniquely
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {number} [options.esrType=Molecule.cESRTypeAnd]
     * @param {boolean} [options.atLeastThreeAtoms=true] - if true, only carbons with at least three atoms will be considered
     */
    function ensureHeterotopicChiralBonds(molecule, options = {}) {
      const {
        Molecule
      } = molecule.getOCL();
      const {
        esrType = Molecule.cESRTypeAnd,
        atLeastThreeAtoms = true
      } = options;
      molecule.ensureHelperArrays(Molecule.cHelperBitNeighbours);
      const heterotopicCarbons = getChiralOrHeterotopicCarbons(molecule);
      for (const i of heterotopicCarbons) {
        if (atLeastThreeAtoms && molecule.getAllConnAtoms(i) < 3) continue;
        if (molecule.getStereoBond(i) === -1) {
          const stereoBond = molecule.getAtomPreferredStereoBond(i);
          if (stereoBond !== -1) {
            molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
            if (molecule.getBondAtom(1, stereoBond) === i) {
              const connAtom = molecule.getBondAtom(0, stereoBond);
              molecule.setBondAtom(0, stereoBond, i);
              molecule.setBondAtom(1, stereoBond, connAtom);
            }
            // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
            molecule.setAtomESR(i, esrType, 0);
          }
        }
      }
    }

    /**
     * Expand all the implicit hydrogens and ensure chiral bonds on heterotopic bonds
     * @param molecule
     * @param options
     * @returns
     */
    function getMoleculeWithH(molecule, options) {
      const {
        logger,
        maxNbAtoms
      } = options;
      const moleculeWithH = molecule.getCompactCopy();
      moleculeWithH.addImplicitHydrogens();
      if (moleculeWithH.getAllAtoms() > maxNbAtoms) {
        logger.warn(`too many atoms to evaluate heterotopic chiral bonds: ${moleculeWithH.getAllAtoms()} > ${maxNbAtoms}`);
      } else {
        ensureHeterotopicChiralBonds(moleculeWithH);
      }
      return moleculeWithH;
    }

    /**
     * In order to be able to give a unique ID to all the atoms we are replacing the H by X
     * @param moleculeWithH
     * @returns
     */
    function getXMolecule(moleculeWithH) {
      const xAtomNumber = getXAtomicNumber(moleculeWithH);
      const xMolecule = moleculeWithH.getCompactCopy();
      for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (xMolecule.getAtomicNo(i) === 1) {
          xMolecule.setAtomicNo(i, xAtomNumber);
        }
      }
      return xMolecule;
    }

    /**
     * This class deals with topicity information and hose codes
     * It is optimized to avoid recalculation of the same information
     */
    class TopicMolecule {
      originalMolecule;
      molecule;
      idCode;
      options;
      cache;
      constructor(molecule, options = {}) {
        this.originalMolecule = molecule;
        this.options = {
          maxPathLength: 5,
          maxNbAtoms: 250,
          logger: console,
          ...options
        };
        this.idCode = molecule.getIDCode();
        this.molecule = this.originalMolecule.getCompactCopy();
        this.molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
        this.molecule.ensureHelperArrays(this.molecule.getOCL().Molecule.cHelperNeighbours);
        this.cache = {};
      }
      /**
       * This method ensures that all the atoms have a mapNo corresponding to the atom number.
       * It will enforce mapNo in molecule and moleculeWithH
       * We start numbering the atoms at 1
       */
      setAtomNoInMapNo() {
        const molecules = [this.molecule, this.moleculeWithH];
        for (const molecule of molecules) {
          for (let i = 0; i < molecule.getAllAtoms(); i++) {
            molecule.setAtomMapNo(i, i + 1, false);
          }
        }
      }
      /**
       * This method ensures that all the atoms have a mapNo in the molecule (and not the moleculeWithH! )
       */
      ensureMapNo() {
        const existingMapNo = {};
        for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
          const mapNo = this.molecule.getAtomMapNo(i);
          if (mapNo) {
            if (existingMapNo[mapNo]) {
              throw new Error('The molecule contains several atoms with the same mapNo');
            }
            existingMapNo[mapNo] = true;
          }
        }
        let nextMapNo = 1;
        for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
          const mapNo = this.molecule.getAtomMapNo(i);
          if (!mapNo) {
            while (existingMapNo[nextMapNo]) {
              nextMapNo++;
            }
            existingMapNo[nextMapNo] = true;
            this.molecule.setAtomMapNo(i, nextMapNo, false);
          }
        }
      }
      /**
       * For each atom we will return an array of objects that contains the
       * different possible path as well as the canonic hose codes
       * @param options
       * @returns
       */
      getHoseCodesForPath(options = {}) {
        return getCanonizedHoseCodesForPath(this, options);
      }
      /**
       * Return one fragment for a specific sphere size and specific root atoms
       * @param rootAtoms
       * @param options
       * @returns
       */
      getHoseFragment(rootAtoms, options = {}) {
        const {
          sphereSize = 2,
          tagAtoms = rootAtoms,
          tagAtomFct
        } = options;
        const fragments = getHoseCodesForAtomsAsFragments(this.moleculeWithH, {
          rootAtoms,
          minSphereSize: sphereSize,
          maxSphereSize: sphereSize,
          tagAtoms,
          tagAtomFct
        });
        return fragments[0];
      }
      getAtomPathsFrom(atom, options = {}) {
        const {
          minPathLength = 1,
          maxPathLength = this.options.maxPathLength,
          toAtomicNo
        } = options;
        if (maxPathLength > this.options.maxPathLength) {
          throw new Error('The maxPathLength is too long, you should increase the maxPathLength when instantiating the TopicMolecule');
        }
        const atomPaths = this.atomsPaths[atom];
        if (!atomPaths) {
          throw new Error('Unexpected missing atom path');
        }
        const paths = [];
        for (let i = minPathLength; i <= maxPathLength; i++) {
          const atomPathValue = atomPaths[i];
          if (!atomPathValue) {
            throw new Error(`Unexpected missing atom path at index ${i}`);
          }
          for (const atomPath of atomPathValue) {
            if (!toAtomicNo || this.moleculeWithH.getAtomicNo(atomPath.path.at(-1)) === toAtomicNo) {
              paths.push(atomPath.path);
            }
          }
        }
        return paths;
      }
      getAtomPaths(atom1, atom2, options = {}) {
        const {
          pathLength
        } = options;
        if (pathLength !== undefined && pathLength > this.options.maxPathLength) {
          throw new Error('The distance is too long, you should increase the maxPathLength when instantiating the TopicMolecule');
        }
        const atomPaths = this.atomsPaths[atom1];
        if (!atomPaths) {
          throw new Error('Unexpected missing atom path');
        }
        const minDistance = pathLength || 0;
        const maxDistance = pathLength || this.options.maxPathLength;
        const paths = [];
        for (let i = minDistance; i <= maxDistance; i++) {
          const atomPathValue = atomPaths[i];
          if (!atomPathValue) {
            throw new Error(`Unexpected missing atom path at index ${i}`);
          }
          for (const atomPath of atomPathValue) {
            if (atomPath.path.at(-1) === atom2) {
              paths.push(atomPath.path);
            }
          }
        }
        return paths;
      }
      get atomsPaths() {
        if (this.cache.atomsPaths) return this.cache.atomsPaths;
        this.cache.atomsPaths = getAllAtomsPaths(this.moleculeWithH, {
          maxPathLength: this.options.maxPathLength
        });
        return this.cache.atomsPaths;
      }
      toMolfile(options = {}) {
        const {
          version = 2
        } = options;
        if (version === 2) {
          return this.molecule.toMolfile();
        }
        return this.molecule.toMolfileV3();
      }
      getMolecule() {
        return this.molecule;
      }
      /**
       * Returns a new TopicMolecule but will copy precalculated information
       * if possible (same idCode). This is very practical when expanding hydrogens
       * for example.
       * @param molecule
       * @returns
       */
      fromMolecule(molecule) {
        const idCode = molecule.getIDCode();
        if (idCode !== this.idCode) {
          // no way for optimisation
          return new TopicMolecule(molecule);
        }
        const topicMolecule = new TopicMolecule(molecule);
        topicMolecule.cache = {
          canonizedDiaIDs: this.cache.canonizedDiaIDs,
          canonizedHoseCodes: this.cache.canonizedHoseCodes
        };
        return topicMolecule;
      }
      /**
       * Returns a molecule with all the hydrogens added. The order is NOT canonized
       */
      get moleculeWithH() {
        if (this.cache.moleculeWithH) return this.cache.moleculeWithH;
        this.cache.moleculeWithH = getMoleculeWithH(this.molecule, {
          maxNbAtoms: this.options.maxNbAtoms,
          logger: this.options.logger
        });
        return this.cache.moleculeWithH;
      }
      get xMolecule() {
        if (this.cache.xMolecule) return this.cache.xMolecule;
        this.cache.xMolecule = getXMolecule(this.moleculeWithH);
        return this.cache.xMolecule;
      }
      /**
       * This is related to the current moleculeWithH. The order is NOT canonized
       */
      get diaIDs() {
        if (this.cache.diaIDs) return this.cache.diaIDs;
        const diaIDs = [];
        if (this.moleculeWithH.getAllAtoms() > this.options.maxNbAtoms) {
          this.options.logger.warn(`too many atoms to evaluate heterotopicity: ${this.moleculeWithH.getAllAtoms()} > ${this.options.maxNbAtoms}`);
        } else {
          for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
            diaIDs.push(this.canonizedDiaIDs[this.finalRanks[i]]);
          }
        }
        this.cache.diaIDs = diaIDs;
        return diaIDs;
      }
      /**
       * We return the atomIDs corresponding to the specified diaID as well has the attached hydrogens or heavy atoms
       * @param diaID
       * @returns
       */
      getDiaIDsObject() {
        return groupDiastereotopicAtomIDsAsObject(this.diaIDs, this.molecule, this.moleculeWithH);
      }
      /**
       * This is related to the current moleculeWithH. The order is NOT canonized
       */
      get hoseCodes() {
        if (this.cache.hoseCodes) return this.cache.hoseCodes;
        const hoseCodes = [];
        for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
          hoseCodes.push(this.canonizedHoseCodes[this.finalRanks[i]]);
        }
        this.cache.hoseCodes = hoseCodes;
        return hoseCodes;
      }
      get canonizedDiaIDs() {
        if (this.cache.canonizedDiaIDs) return this.cache.canonizedDiaIDs;
        this.cache.canonizedDiaIDs = getCanonizedDiaIDs(this, {
          maxNbAtoms: this.options.maxNbAtoms,
          logger: this.options.logger
        });
        return this.cache.canonizedDiaIDs;
      }
      get canonizedHoseCodes() {
        if (this.cache.canonizedHoseCodes) {
          return this.cache.canonizedHoseCodes;
        }
        this.cache.canonizedHoseCodes = getCanonizedHoseCodes(this);
        return this.cache.canonizedHoseCodes;
      }
      /**
       * Returns the distance matrix for the current moleculeWithH
       */
      get distanceMatrix() {
        return getConnectivityMatrix(this.moleculeWithH, {
          pathLength: true
        });
      }
      get diaIDsAndInfo() {
        if (this.cache.diaIDsAndInfo) {
          return this.cache.diaIDsAndInfo;
        }
        this.cache.diaIDsAndInfo = getDiaIDsAndInfo(this, this.canonizedDiaIDs);
        return this.cache.diaIDsAndInfo;
      }
      /**
       * Returns symmetryRanks for all the atoms including hydrogens. Those ranks
       * deals with topicity and is related to the current moleculeWithH.
       * In order to calculate the ranks we replace all the
       * hydrogens with a X atom.
       */
      get heterotopicSymmetryRanks() {
        if (this.cache.heterotopicSymmetryRanks) {
          return this.cache.heterotopicSymmetryRanks;
        }
        this.cache.heterotopicSymmetryRanks = getHeterotopicSymmetryRanks(this.xMolecule);
        return [...this.cache.heterotopicSymmetryRanks];
      }
      /**
       * Returns finalRanks for all the atoms including hydrogens. Those ranks
       * deals with topicity and is related to the current moleculeWithH.
       * All the atoms have a unique identifier.j
       * In order to calculate the ranks we replace all the
       * hydrogens with a X atom.
       */
      get finalRanks() {
        if (this.cache.finalRanks) return this.cache.finalRanks;
        this.cache.finalRanks = getFinalRanks(this.xMolecule);
        return this.cache.finalRanks;
      }
      toMolfileWithH(options = {}) {
        const {
          version = 2
        } = options;
        if (version === 2) {
          return this.moleculeWithH.toMolfile();
        }
        return this.moleculeWithH.toMolfileV3();
      }
      toMolfileWithoutH(options = {}) {
        const molecule = this.molecule.getCompactCopy();
        molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
        const atomsToDelete = [];
        for (let atomID = 0; atomID < molecule.getAtoms(); atomID++) {
          if (molecule.getExplicitHydrogens(atomID) > 0) {
            for (let i = 0; i < molecule.getAllConnAtoms(atomID); i++) {
              const connectedAtom = molecule.getConnAtom(atomID, i);
              if (molecule.getAtomicNo(connectedAtom) === 1) {
                atomsToDelete.push(connectedAtom);
              }
            }
          }
        }
        molecule.deleteAtoms(atomsToDelete);
        const {
          version = 2
        } = options;
        if (version === 2) {
          return molecule.toMolfile();
        }
        return molecule.toMolfileV3();
      }
      /**
       * Returns an array of objects containing the oclID and the corresponding hydrogens and atoms
       * for the specified atomLabel (if any)
       * This always applied to the molecule with expanded hydrogens and chirality
       * @param options
       * @returns
       */
      getGroupedDiastereotopicAtomIDs(options = {}) {
        if (!this.diaIDs) return undefined;
        return groupDiastereotopicAtomIDs$1(this.diaIDs, this.moleculeWithH, options);
      }
      /**
       * This method returns a mapping between the diaIDs of the current molecule.
       * It expects that the initial molfile and the final molfile contains atomMapNo
       * in order to track which atom becomes which one.
       * @param molecule
       */
      getDiaIDsMapping(molecule) {
        const topicMolecule = new TopicMolecule(molecule);
        const originalDiaIDs = this.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
        const destinationDiaIDs = topicMolecule.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
        const mapping = {};
        // we first check all the atoms present in the molfile
        for (const destinationDiaID of destinationDiaIDs) {
          const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
          const newIDCode = destinationDiaID.idCode;
          const oldIDCode = originalDiaID.idCode;
          if (oldIDCode in mapping) {
            if (mapping[oldIDCode] !== newIDCode) {
              mapping[oldIDCode] = undefined;
            }
          } else {
            mapping[oldIDCode] = newIDCode;
          }
        }
        // we now check all the attached hydrogens that are not defined in the molfile and were not yet mapped
        for (const destinationDiaID of destinationDiaIDs) {
          const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
          for (let i = 0; i < originalDiaID.attachedHydrogensIDCodes.length; i++) {
            const oldHydrogenIDCode = originalDiaID.attachedHydrogensIDCodes.at(i);
            if (mapping[oldHydrogenIDCode]) continue;
            const newHydrogenIDCode = destinationDiaID.attachedHydrogensIDCodes[i];
            if (oldHydrogenIDCode && newHydrogenIDCode) {
              if (oldHydrogenIDCode in mapping) {
                if (mapping[oldHydrogenIDCode] !== newHydrogenIDCode) {
                  mapping[oldHydrogenIDCode] = undefined;
                }
              } else {
                mapping[oldHydrogenIDCode] = newHydrogenIDCode;
              }
            }
          }
        }
        return mapping;
      }
    }
    function groupDiastereotopicAtomIDs$1(diaIDs, molecule, options = {}) {
      const diaIDsObject = groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, molecule, options);
      return Object.values(diaIDsObject);
    }
    function groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, moleculeWithH, options = {}) {
      const {
        atomLabel
      } = options;
      const diaIDsObject = {};
      for (let i = 0; i < diaIDs.length; i++) {
        if (!atomLabel || moleculeWithH.getAtomLabel(i) === atomLabel) {
          const diaID = diaIDs[i];
          if (!diaIDsObject[diaID]) {
            diaIDsObject[diaID] = {
              counter: 0,
              oclID: diaID,
              atomLabel: moleculeWithH.getAtomLabel(i),
              atoms: [],
              existingAtoms: [],
              heavyAtoms: [],
              attachedHydrogens: [],
              customLabels: [],
              attachedHydrogensCustomLabels: [],
              heavyAtomsCustomLabels: []
            };
          }
          const customLabel = moleculeWithH.getAtomCustomLabel(i);
          if (customLabel && !diaIDsObject[diaID].customLabels.includes(customLabel)) {
            diaIDsObject[diaID].customLabels.push(customLabel);
          }
          if (moleculeWithH.getAtomicNo(i) === 1) {
            const connected = moleculeWithH.getConnAtom(i, 0);
            if (!diaIDsObject[diaID].heavyAtoms.includes(connected)) {
              diaIDsObject[diaID].heavyAtoms.push(connected);
            }
            const heavyAtomCustomLabel = molecule.getAtomCustomLabel(connected);
            if (heavyAtomCustomLabel && !diaIDsObject[diaID].heavyAtomsCustomLabels.includes(heavyAtomCustomLabel)) {
              diaIDsObject[diaID].heavyAtomsCustomLabels.push(heavyAtomCustomLabel);
            }
            if (molecule.getAtomicNo(i)) {
              diaIDsObject[diaID].existingAtoms.push(i);
            } else if (!diaIDsObject[diaID].existingAtoms.includes(connected)) {
              diaIDsObject[diaID].existingAtoms.push(connected);
            }
          } else {
            for (let j = 0; j < moleculeWithH.getAllConnAtoms(i); j++) {
              const connected = moleculeWithH.getConnAtom(i, j);
              if (moleculeWithH.getAtomicNo(connected) === 1) {
                const attachedHydrogenCustomLabel = molecule.getAtomCustomLabel(connected);
                if (attachedHydrogenCustomLabel && !diaIDsObject[diaID].attachedHydrogensCustomLabels.includes(attachedHydrogenCustomLabel)) {
                  diaIDsObject[diaID].attachedHydrogensCustomLabels.push(attachedHydrogenCustomLabel);
                }
                diaIDsObject[diaID].attachedHydrogens.push(connected);
              }
            }
            diaIDsObject[diaID].existingAtoms.push(i);
          }
          diaIDsObject[diaID].counter++;
          diaIDsObject[diaID].atoms.push(i);
        }
      }
      for (const diaIDObject of Object.values(diaIDsObject)) {
        diaIDObject.existingAtoms.sort((a, b) => a - b);
        diaIDObject.attachedHydrogens.sort((a, b) => a - b);
      }
      return diaIDsObject;
    }

    /**
     * Returns an array of symmetry ranks.
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @returns
     */
    function getSymmetryRanks(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      const xAtomNumber = getXAtomicNumber(molecule);
      // most of the molecules have some symmetry
      const internalMolecule = molecule.getCompactCopy();
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (internalMolecule.getAtomicNo(i) === 1) {
          internalMolecule.setAtomicNo(i, xAtomNumber);
        }
      }
      internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const symmetryRanks = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        symmetryRanks.push(internalMolecule.getSymmetryRank(i));
      }
      return symmetryRanks;
    }

    /**
     * Returns an array of diastereotopic ID (as idCode)
     * @param {import('openchemlib').Molecule} molecule
     */
    function getDiastereotopicAtomIDs(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      ensureHeterotopicChiralBonds(molecule);
      const symmetryRanks = getSymmetryRanks(molecule);
      const numberAtoms = molecule.getAllAtoms();
      const ids = [];
      const cache = {};
      for (let iAtom = 0; iAtom < numberAtoms; iAtom++) {
        const rank = symmetryRanks[iAtom];
        if (rank && cache[rank]) {
          ids[iAtom] = cache[rank];
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(molecule);
        tagAtom(tempMolecule, iAtom);
        makeRacemic(tempMolecule);
        // We need to ensure the helper array in order to get correctly the result of racemisation
        ids[iAtom] = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        cache[rank] = ids[iAtom];
      }
      return ids;
    }

    /**
     *
     * @param {import('openchemlib').Molecule} molecule
     */
    function getDiastereotopicAtomIDsAndH(molecule) {
      const OCL = molecule.getOCL();
      molecule = molecule.getCompactCopy();
      molecule.addImplicitHydrogens();
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      const diaIDs = getDiastereotopicAtomIDs(molecule);
      const newDiaIDs = [];
      for (let i = 0; i < diaIDs.length; i++) {
        const diaID = diaIDs[i];
        const newDiaID = {
          oclID: diaID,
          hydrogenOCLIDs: [],
          nbHydrogens: 0
        };
        if (molecule.getAtomicNo(i) === 1) {
          const atom = molecule.getConnAtom(i, 0);
          newDiaID.heavyAtom = diaIDs[atom];
        }
        for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
          const atom = molecule.getConnAtom(i, j);
          if (molecule.getAtomicNo(atom) === 1) {
            newDiaID.nbHydrogens++;
            if (!newDiaID.hydrogenOCLIDs.includes(diaIDs[atom])) {
              newDiaID.hydrogenOCLIDs.push(diaIDs[atom]);
            }
          }
        }
        newDiaIDs.push(newDiaID);
      }
      return newDiaIDs;
    }

    function groupDiastereotopicAtomIDs(diaIDs, molecule, options = {}) {
      const {
        atomLabel
      } = options;
      const diaIDsObject = {};
      for (let i = 0; i < diaIDs.length; i++) {
        if (!atomLabel || molecule.getAtomLabel(i) === atomLabel) {
          const diaID = diaIDs[i];
          if (!diaIDsObject[diaID]) {
            diaIDsObject[diaID] = {
              counter: 0,
              atoms: [],
              oclID: diaID,
              atomLabel: molecule.getAtomLabel(i)
            };
          }
          diaIDsObject[diaID].counter++;
          diaIDsObject[diaID].atoms.push(i);
        }
      }
      return Object.keys(diaIDsObject).map(key => diaIDsObject[key]);
    }

    /**
     * This function groups the diasterotopic atomIds of the molecule based on equivalence of atoms. The output object contains
     * a set of chemically equivalent atoms(element.atoms) and the groups of magnetically equivalent atoms (element.magneticGroups)
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {string} [options.atomLabel] - Select atoms of the given atomLabel. By default it returns all the explicit atoms in the molecule
     * @returns {Array}
     */
    function getGroupedDiastereotopicAtomIDs(molecule, options = {}) {
      const diaIDs = getDiastereotopicAtomIDs(molecule);
      return groupDiastereotopicAtomIDs(diaIDs, molecule, options);
    }

    /**
     * Parse a molfile and returns an object containing the molecule, the map and the diaIDs
     * The map allows to reload properties assigned to the atom molfile
     * Please take care than numbering of atoms starts at 0 !
     * @param {typeof import('openchemlib')} OCL - openchemlib library
     * @param {string} molfile
     * @returns
     */
    function getDiastereotopicAtomIDsFromMolfile(OCL, molfile) {
      const {
        map,
        molecule
      } = OCL.Molecule.fromMolfileWithAtomMap(molfile);
      const diaIDsArray = getDiastereotopicAtomIDsAndH(molecule);
      const diaIDs = {};
      for (let i = 0; i < map.length; i++) {
        diaIDs[map[i]] = {
          source: map[i],
          destination: i,
          ...diaIDsArray[i]
        };
      }
      return {
        map: diaIDs,
        molecule,
        diaIDs: diaIDsArray
      };
    }

    /**
     * Returns a SVG
     * @param {*} molecule
     * @param {*} [options={}]
     */
    function toDiastereotopicSVG(molecule, options = {}) {
      const {
        width = 300,
        height = 200,
        prefix = 'ocl',
        heavyAtomHydrogen = false
      } = options;
      let svg = options.svg;
      let diaIDs = [];
      const hydrogenInfo = {};
      for (const line of getDiastereotopicAtomIDsAndH(molecule)) {
        hydrogenInfo[line.oclID] = line;
      }
      if (heavyAtomHydrogen) {
        for (let i = 0; i < molecule.getAtoms(); i++) {
          diaIDs.push([]);
        }
        const groupedDiaIDs = molecule.getGroupedDiastereotopicAtomIDs();
        for (const diaID of groupedDiaIDs) {
          if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
            for (const atom of diaID.atoms) {
              for (const id of hydrogenInfo[diaID.oclID].hydrogenOCLIDs) {
                if (!diaIDs[atom * 1].includes(id)) diaIDs[atom].push(id);
              }
            }
          }
        }
      } else {
        diaIDs = molecule.getDiastereotopicAtomIDs().map(a => [a]);
      }
      if (!svg) svg = molecule.toSVG(width, height, prefix);
      svg = svg.replaceAll(/Atom:\d+"/g, value => {
        const atom = value.replaceAll(/\D/g, '');
        return `${value} data-diaid="${diaIDs[atom].join(',')}"`;
      });
      return svg;
    }

    /**
     * Returns the hose codes for all atoms in the molecule
     * @param {*} molecule
     * @param {object} [options={}]
     * @param {string[]} [options.atomLabels]
     * @param {number} [options.minSphereSize=0]
     * @param {number} [options.maxSphereSize=4]
     * @returns
     */
    function getHoseCodes(molecule, options = {}) {
      const {
        atomLabels,
        minSphereSize,
        maxSphereSize
      } = options;
      const {
        Molecule
      } = molecule.getOCL();
      const atomicNumbers = atomLabels?.map(label => Molecule.getAtomicNoFromLabel(label));
      const internalMolecule = getCompactCopyWithoutCustomLabels(molecule);
      internalMolecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(internalMolecule);
      const hoses = [];
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        if (atomicNumbers && !atomicNumbers.includes(internalMolecule.getAtomicNo(i))) {
          hoses.push(undefined);
        } else {
          const tempMolecule = internalMolecule.getCompactCopy();
          tagAtom(tempMolecule, i);
          hoses.push(getHoseCodesForAtomsAsStrings(tempMolecule, {
            minSphereSize,
            maxSphereSize
          }));
        }
      }
      return hoses;
    }

    /**
     * Returns an object containing a molfile, molfile with hydrogens, hoses codes and optionally the diaIDs
     * and the diaIDs
     * The map allows to reload properties assigned to the atom molfile
     * Please take care than numbering of atoms starts at 0 !
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {string[]} [options.atomLabels]
     * @param {boolean} [options.calculateDiastereotopicIDs=true]
     * @param {number} [options.minSphereSize=0]
     * @param {number} [options.maxSphereSize=4]
     * @returns
     */
    function getHoseCodesAndInfo(molecule, options = {}) {
      const {
        minSphereSize,
        maxSphereSize,
        calculateDiastereotopicIDs = true
      } = options;
      const {
        Molecule
      } = molecule.getOCL();
      molecule = molecule.getCompactCopy();
      // this will force reordering of the hydrogens to the end, just to have the same order as in the molfile
      molecule.ensureHelperArrays(Molecule.cHelperNeighbours);
      const newMolfile = molecule.toMolfile();
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const newMolfileWithH = molecule.toMolfile();
      const symmetryRanks = getSymmetryRanks(molecule);
      const cache = {};
      const hoses = [];
      const diaIDs = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        const rank = symmetryRanks[i];
        if (rank && cache[rank]) {
          diaIDs.push(cache[rank].diaID);
          hoses.push(cache[rank].hose);
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(molecule);
        tagAtom(tempMolecule, i);
        let diaID;
        if (calculateDiastereotopicIDs) {
          makeRacemic(tempMolecule);
          diaID = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
          diaIDs.push(diaID);
        }
        const hose = getHoseCodesForAtomsAsStrings(tempMolecule, {
          minSphereSize,
          maxSphereSize
        });
        hoses.push(hose);
        cache[rank] = {
          diaID,
          hose
        };
      }
      const distanceMatrix = getConnectivityMatrix(molecule, {
        pathLength: true
      });
      return {
        molfile: newMolfile,
        molfileWithH: newMolfileWithH,
        hoses,
        diaIDs: calculateDiastereotopicIDs ? diaIDs : undefined,
        moleculeWithHydrogens: molecule,
        distanceMatrix
      };
    }

    const FULL_HOSE_CODE = 1;
    const HOSE_CODE_CUT_C_SP3_SP3 = 2;
    /**
     * Returns the hose code for specific atom numbers
     * @param {import('openchemlib').Molecule} originalMolecule - The OCL molecule to be fragmented
     * @param {number[]} rootAtoms
     * @param {object} [options={}]
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtoms(originalMolecule, rootAtoms = [], options = {}) {
      const {
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE
      } = options;
      const molecule = getCompactCopyWithoutCustomLabels(originalMolecule);
      // those 2 lines should be done only once
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      const allowedCustomLabels = [];
      for (const rootAtom of rootAtoms) {
        allowedCustomLabels.push(tagAtom(molecule, rootAtom));
      }
      return getHoseCodesForAtomsAsStrings(molecule, {
        minSphereSize,
        maxSphereSize,
        allowedCustomLabels,
        kind
      });
    }

    /**
     * Returns the hose code for a specific atom number
     * @param {import('openchemlib').Molecule} originalMolecule
     * @param {number} rootAtom
     * @param {object} [options={}]
     * @param {boolean} [options.isTagged] - Specify is the atom is already tagged
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtom(originalMolecule, rootAtom, options = {}) {
      return getHoseCodesForAtoms(originalMolecule, [rootAtom], options);
    }

    /**
     * Returns an array of strings (idCodes) specified molecule. Each string corresponds to a
     * hose code. By default it will calculate the hose codes for sphere 0 to 4 and will reuse
     * the existing tagged atoms.
     * This method ensure implicity hydrogens and possible missing chiral bonds.
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to process.
     * @param {object} options - Options for generating hose codes.
     * @param {string[]} [options.allowedCustomLabels] - Array of the custom labels that are considered as root atoms. By default all atoms having a customLabel
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     *  @param {number[]} [options.rootAtoms=[]] - Array of atom from which we should start to create the HOSE. By default we will used the taggedAtoms
     * @param {number[]} [options.tagAtoms=[]] - Array of atom indices to tag as root atoms
     * @param {Function} [options.tagAtomFct=tagAtom] - Function to tag an atom as root atom. By default it is defined internal
     * @returns {Array} - An array of hose code fragments.
     */
    function getHoseCodesFromDiastereotopicID(molecule, options = {}) {
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      return getHoseCodesForAtomsAsStrings(molecule, options);
    }

    let fragment$1;
    /**
     * Returns the hose code for a specific atom number
     * @param {import('openchemlib').Molecule} molecule
     * @param from
     * @param to
     * @param maxLength
     */
    function getHoseCodesForPath(molecule, from, to, maxLength) {
      const OCL = molecule.getOCL();
      const originalFrom = from;
      const originalTo = to;
      molecule = getCompactCopyWithoutCustomLabels(molecule);
      const originalAtoms = []; // path before renumbering
      molecule.getPath(originalAtoms, from, to, maxLength + 1);
      let torsion;
      if (originalAtoms.length === 4) {
        torsion = molecule.calculateTorsion(originalAtoms);
      }
      const tag1 = tagAtom(molecule, from);
      const tag2 = tagAtom(molecule, to);
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      from = -1;
      to = -1;
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (tag1 === tag2) {
          if (molecule.getAtomCustomLabel(i) === tag1) {
            if (from === -1) {
              from = i;
            } else {
              to = i;
            }
          }
        } else {
          if (tag1 === molecule.getAtomCustomLabel(i)) {
            from = i;
          }
          if (tag2 === molecule.getAtomCustomLabel(i)) {
            to = i;
          }
        }
      }
      if (!fragment$1) fragment$1 = new OCL.Molecule(0, 0);
      const atoms = [];
      molecule.getPath(atoms, from, to, maxLength + 1);
      let min = 0;
      let max = 0;
      const atomMask = new Array(molecule.getAllAtoms()).fill(false);
      const atomList = new Array(molecule.getAllAtoms()).fill(-1);
      const hoses = [];
      for (let sphere = 0; sphere <= 2; sphere++) {
        if (max === 0) {
          for (const atom of atoms) {
            atomMask[atom] = true;
            atomList[max++] = atom;
          }
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            const atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              const connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                atomMask[connAtom] = true;
                atomList[newMax++] = connAtom;
              }
            }
          }
          min = max;
          max = newMax;
        }
        const atomMap = [];
        molecule.copyMoleculeByAtoms(fragment$1, atomMask, true, atomMap);
        makeRacemic(fragment$1);
        const oclID = fragment$1.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        hoses.push({
          sphere,
          oclID
        });
      }
      return {
        atoms: originalAtoms,
        from: originalFrom,
        to: originalTo,
        torsion,
        hoses,
        length: originalAtoms.length - 1
      };
    }

    /**
     * Create a polymer from a unit, alpha and omega groups
     * A unit must contain a R1 and a R2
     * An alpha end group must contain a R1
     * An omega end group must contain a R2
     * @param {import('openchemlib').Molecule} unit - an instance of OCL.Molecule
     * @param {object} options
     * @param {number} [options.count=10] - number of units
     * @param {boolean} [options.markMonomer=false] - mark the different units of the polymer in the atom map
     * @param {import('openchemlib').Molecule} [options.alpha] - alpha end group, default is an hydrogen
     * @param {import('openchemlib').Molecule} [options.gamma] - omega end group, default is an hydrogen
     */
    function createPolymer(unit, options = {}) {
      const {
        count = 10
      } = options;
      checkEntity(unit, 'unit');
      const {
        Molecule
      } = unit.getOCL();
      const {
        alpha,
        gamma
      } = getAlphaGamma(unit, options);
      checkEntity(alpha, 'alpha');
      checkEntity(gamma, 'gamma');
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      const polymer = alpha.getCompactCopy();
      polymer.addMolecule(getUnit(unit, 1, options));
      addBond(polymer, r1AtomicNo, r1AtomicNo);
      for (let i = 0; i < count - 1; i++) {
        polymer.addMolecule(getUnit(unit, i + 2, options));
        addBond(polymer, r2AtomicNo, r1AtomicNo);
      }
      polymer.addMolecule(gamma);
      addBond(polymer, r2AtomicNo, r2AtomicNo);
      polymer.ensureHelperArrays(Molecule.cHelperNeighbours);
      // encoding directly in atomNapNo didn't work out because it was removed when deleting atoms
      for (let i = 0; i < polymer.getAtoms(); i++) {
        polymer.setAtomMapNo(i, (polymer.getAtomCustomLabel(i) || '').replace(/monomer_/, ''));
        polymer.setAtomCustomLabel(i, '');
      }
      return polymer;
    }
    function getUnit(unit, index, options) {
      const {
        markMonomer = false
      } = options;
      if (markMonomer) {
        unit = unit.getCompactCopy();
        unit.ensureHelperArrays(unit.getOCL().Molecule.cHelperNeighbours);
        for (let j = 0; j < unit.getAtoms(); j++) {
          unit.setAtomCustomLabel(j, `monomer_${index}`);
        }
      }
      return unit;
    }
    function addBond(molecule, firstAtomicNo, secondAtomicNo) {
      molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
      let i, j;
      loop: for (i = 0; i < molecule.getAtoms(); i++) {
        if (molecule.getAtomicNo(i) === firstAtomicNo) {
          for (j = i + 1; j < molecule.getAtoms(); j++) {
            if (molecule.getAtomicNo(j) === secondAtomicNo) {
              molecule.addBond(molecule.getConnAtom(i, 0), molecule.getConnAtom(j, 0), 1);
              break loop;
            }
          }
        }
      }
      molecule.deleteAtoms([i, j]);
    }
    function checkEntity(unit, kind) {
      let nbR1 = 1;
      let nbR2 = 1;
      switch (kind) {
        case 'unit':
          break;
        case 'alpha':
          nbR2 = 0;
          break;
        case 'gamma':
          nbR1 = 0;
          break;
        default:
          throw new Error('Unknown kind');
      }
      if (!unit) {
        throw new Error('unit is required');
      }
      const {
        Molecule
      } = unit.getOCL();
      // unit must contain ONE R1 and ONE R2
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      let r1Count = 0;
      let r2Count = 0;
      for (let i = 0; i < unit.getAtoms(); i++) {
        if (unit.getAtomicNo(i) === r1AtomicNo) {
          r1Count++;
        }
        if (unit.getAtomicNo(i) === r2AtomicNo) {
          r2Count++;
        }
      }
      if (r1Count !== nbR1) {
        throw new Error(`${kind} must contain ${nbR1} R1`);
      }
      if (r2Count !== nbR2) {
        throw new Error(`${kind} must contain ${nbR2} R2`);
      }
    }
    function getAlphaGamma(unit, options) {
      let {
        alpha,
        gamma
      } = options;
      const {
        Molecule
      } = unit.getOCL();
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      if (!alpha) {
        alpha = Molecule.fromSmiles('CC');
        alpha.setAtomicNo(0, r1AtomicNo);
        alpha.setAtomicNo(1, 1);
      }
      if (!gamma) {
        gamma = Molecule.fromSmiles('CC');
        gamma.setAtomicNo(0, r2AtomicNo);
        gamma.setAtomicNo(1, 1);
      }
      return {
        alpha,
        gamma
      };
    }
    function getR1R2AtomicNo(Molecule) {
      const r1AtomicNo = Molecule.getAtomicNoFromLabel('R1', Molecule.cPseudoAtomsRGroups);
      const r2AtomicNo = Molecule.getAtomicNoFromLabel('R2', Molecule.cPseudoAtomsRGroups);
      return {
        r1AtomicNo,
        r2AtomicNo
      };
    }

    var NP=class{copyMolecule(ft){let vt=ft.getIDCodeAndCoordinates();navigator.clipboard.writeText(`${vt.idCode} ${vt.coordinates}`);}pasteMolecule(){return null;}};var hu="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",En=new Uint8Array(256);for(let d=0;d<hu.length;d++)En[hu.charCodeAt(d)]=d;function Tu(d){let ft=d.length*.75,vt=d.length,kt,Ot=0,p1,zt,t1,at;d.at(-1)==="="&&(ft--,d.at(-2)==="="&&ft--);let g1=new ArrayBuffer(ft),T1=new Uint8Array(g1);for(kt=0;kt<vt;kt+=4)p1=En[d.charCodeAt(kt)],zt=En[d.charCodeAt(kt+1)],t1=En[d.charCodeAt(kt+2)],at=En[d.charCodeAt(kt+3)],T1[Ot++]=p1<<2|zt>>4,T1[Ot++]=(zt&15)<<4|t1>>2,T1[Ot++]=(t1&3)<<6|at&63;return g1;}function IP(d){return d.toString(16).padStart(2,"0");}var _A=globalThis.devicePixelRatio||1;var We=class{constructor(ft){this.ctx=ft,this.ctx.textAlign="left",this.ctx.textBaseline="top",this.currentFontSize=12,this.currentFont="12px sans-serif",this.ctx.font=this.currentFont,this.currentColor="#000000",this.currentLineWidth=1,this.canvasCache=new Map();}clearRect(ft,vt,kt,Ot){this.ctx.clearRect(ft,vt,kt,Ot);}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}getSelectionBackgroundRGB(){return 12310268;}getLineWidth(){return this.currentLineWidth;}setRGB(ft){let vt=ft>>>16&255,kt=ft>>>8&255,Ot=ft>>>0&255;this.currentColor=`#${IP(vt)}${IP(kt)}${IP(Ot)}`,this.ctx.fillStyle=this.currentColor,this.ctx.strokeStyle=this.currentColor;}setFont(ft,vt,kt){this.currentFontSize=ft,this.currentFont=`${vt?"bold":""} ${kt?"italic":""} ${ft}px sans-serif`,this.ctx.font=this.currentFont;}getFontSize(){return this.currentFontSize;}getBounds(ft){let vt=this.ctx.measureText(ft);return {x:vt.actualBoundingBoxLeft,y:vt.actualBoundingBoxAscent,width:vt.actualBoundingBoxRight,height:vt.actualBoundingBoxAscent};}drawString(ft,vt,kt){this.ctx.fillText(kt,ft,vt);}drawCenteredString(ft,vt,kt){this.ctx.textAlign="center",this.ctx.textBaseline="middle",this.ctx.fillText(kt,ft,vt),this.ctx.textAlign="left",this.ctx.textBaseline="top";}setLineWidth(ft){this.currentLineWidth=ft,this.ctx.lineWidth=ft;}fillRectangle(ft,vt,kt,Ot){this.ctx.fillRect(ft,vt,kt,Ot);}fillCircle(ft,vt,kt){let Ot=kt/2;this.ctx.beginPath(),this.ctx.arc(ft+Ot,vt+Ot,Ot,0,2*Math.PI),this.ctx.fill();}drawLine(ft,vt,kt,Ot){this.ctx.beginPath(),this.ctx.moveTo(ft,vt),this.ctx.lineTo(kt,Ot),this.ctx.stroke();}drawPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let vt=1;vt<ft.getSize();vt++)this.ctx.lineTo(ft.getX(vt),ft.getY(vt));this.ctx.stroke();}drawRectangle(ft,vt,kt,Ot){this.ctx.strokeRect(ft,vt,kt,Ot);}fillPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let vt=1;vt<ft.getSize();vt++)this.ctx.lineTo(ft.getX(vt),ft.getY(vt));this.ctx.fill();}drawImage(ft,vt,kt,Ot,p1,zt,t1,at,g1){if(arguments.length!==9)throw new Error(`drawImage call with ${arguments.length} arguments unimplemented`);let T1=this.canvasCache.get(ft);if(!T1){T1=document.createElement("canvas");let k1=ft.imageData;T1.width=k1.width,T1.height=k1.height;let i1=T1.getContext("2d");i1.globalAlpha=0,i1.putImageData(k1,0,0),this.canvasCache.set(ft,T1);}this.ctx.drawImage(T1,vt,kt,Ot,p1,zt,t1,at,g1);}isDarkBackground(){return  false;}};var OP=class{constructor(ft,vt){this.canvasElement=ft,this.changeListener=vt,this.drawContext=new We(this.canvasElement.getContext("2d"));}getBackgroundRGB(){return 16777215;}getCanvasWidth(){return this.canvasElement.width;}getCanvasHeight(){return this.canvasElement.height;}getDrawContext(){return this.drawContext;}onChange(ft,vt){this.changeListener?.({what:ft,isUserEvent:vt});}getClipboardHandler(){return new NP();}};var f2=`
/* We can customize editor styles here. */
`,M8;function bu(){if(M8)return M8;let d=new CSSStyleSheet();return d.replaceSync(f2),M8=d,d;}function H8(d,ft,vt){let kt=-1;function Ot(zt,t1,at=0){t1.button>0||ft.fireMouseEvent(zt,t1.button+1,at,Math.round(t1.offsetX*_A),Math.round(t1.offsetY*_A),t1.shiftKey,t1.ctrlKey,t1.altKey,t1.button===2);}d.addEventListener("pointerdown",zt=>{kt===-1&&(kt=zt.pointerId,Ot(vt.MOUSE_EVENT_PRESSED,zt));});function p1(zt){kt===zt.pointerId&&(kt=-1,Ot(vt.MOUSE_EVENT_RELEASED,zt));}return document.addEventListener("pointerup",p1),d.addEventListener("click",zt=>{Ot(vt.MOUSE_EVENT_CLICKED,zt,zt.detail);}),d.addEventListener("pointerenter",zt=>{Ot(vt.MOUSE_EVENT_ENTERED,zt);}),d.addEventListener("pointerleave",zt=>{Ot(vt.MOUSE_EVENT_EXITED,zt);}),d.addEventListener("pointermove",zt=>{kt!==-1?kt===zt.pointerId&&Ot(vt.MOUSE_EVENT_DRAGGED,zt):Ot(vt.MOUSE_EVENT_MOVED,zt);}),()=>{document.removeEventListener("pointerup",p1);};}function cu(d,ft,vt,kt,Ot){let p1=typeof navigator<"u"&&navigator.platform==="MacIntel",zt=at=>p1&&at.metaKey||!p1&&at.ctrlKey;function t1(at,g1){let T1=D2(g1,kt);T1!==null&&vt.fireKeyEvent(at,T1,g1.altKey,g1.ctrlKey,g1.shiftKey,zt(g1));}return ft.addEventListener("keydown",at=>{zt(at)&&at.key==="c"||zt(at)&&at.key==="v"||t1(kt.KEY_EVENT_PRESSED,at);}),ft.addEventListener("keyup",at=>{t1(kt.KEY_EVENT_RELEASED,at);}),d.addEventListener("paste",at=>{let g1=at.clipboardData.getData("text"),T1=Ot.fromText(g1);T1&&T1.getAllAtoms()>0&&vt.addPastedOrDropped(T1);}),()=>{};}function D2(d,ft){switch(d.key){case "Control":return ft.KEY_CTRL;case "Alt":return ft.KEY_ALT;case "Shift":return ft.KEY_SHIFT;case "Delete":case "Backspace":return ft.KEY_DELETE;case "F1":return ft.KEY_HELP;case "Escape":return ft.KEY_ESCAPE;case "Enter":return ft.KEY_ENTER;default:return d.key.length===1?d.key.codePointAt(0):null;}}var GP=class{constructor(ft){this.canvasElement=ft,this.drawContext=new We(this.canvasElement.getContext("2d"));}setDimensions(ft,vt){this.canvasElement.width=ft,this.canvasElement.style.width=`${ft/_A}px`,this.canvasElement.height=vt,this.canvasElement.style.height=`${vt/_A}px`;}getDrawContext(){return this.drawContext;}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}};var wu={"chain.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB2UlEQVR4XmNgGJmAkYGJgQULZGXAhCygIGICQg4GdjTIARRDh5xAEQwNKYzXeK4xY1Oez3SNfz8TmoYUxq+6m9bfqLjDI4tmfhzLff9VR17FoWhwZXpkOm2v1C+79+cbD4gga8lneuhrfdv6/+ItSBpSGB+bTDos9IfzP9N/k3cX6+7wwByWz/TWR/E1w/+kO+8j4BpSGL9rz9wh+Efs14TF3heY/tu/vlF2hxtkSxzLY2/FVyz/om69jT0G8kM6QyNDM9MT41nbZX9y/J/ct1/4iU7aIYb/9q8uVx0UbmF64mV7i/1/0s0HSVM5dRgYrjB84/8U+bh80iH+34z/17Xc51ZhUGX8qpkK1GL6+krltyClVyz/c+69C57CwQsK1v+Sl9rdvgr8Z/7P9WvKxNs8fsD44ABqeWLkdYHhv9sL8e+s/xJuv4k4yajEAI6H/+6llxn+g6DUh75gSXhgugEdmXIUJB5150HSFA59oHKIBteGSyBhpv+q9wsVQaZDICdDKuNnneRjCY9exU3l5AUrh2gQelAQ/8Tgve2Tb25NjIgYBkmnMl5Sumi+hQmiGKqhgaGB57Pts4Bfek0sjkjmIxQhs8BpiZFBh0EPCLGlTnQxcGod8gAAp9es6fCW8G8AAAAASUVORK5CYII=)","eraser.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAC1UlEQVR4XnWU309SYRjHn8OYTTM3zTmjC8NFFBR10dJu27wx11VX/QNdeGebTsvua93mRbUmLhURzYgKRZRJUJLibLPpRDCZC3/b1AMc4Tw97+EAR2e8FzD4fr7v53nPOQDCDkT/s2KAKiwROOXvFP8JLnDDGHhgArzgg28wCQGYgiDMw6E+1pY2rsECLEIIIrAC1O2CV/AWzPAOLGCFQbDDJ/hC+CQktU5bXWJoMGlaVv0iPAy/GeCmeBf0QB/YYECKO6kkzAl6Z/+t+BmsSJotgmmN4lFYZcA4tfdQez+1D8Fnio+S1oHe5TAcFuJHVBNit4nGLYrHGOCBbin+ntodMEzxKMkMWy8LpfgSEUN4niC7TTBtqzYYMCG3Z9xHadgD/bC1hi+U4uw1TkhFsqs3adpngJdGHSAZB8VHYAZ4vdt+RSjED3KcvYWYWMJuFY0E+MjdQe5MZgYE7YhFL8soCJzDU1iKwdcE+HOjBoG/NNJbq5DJI40EPNiIPCLgO/Wzk4lS3D2kPyaTQe5hETZsLDfxRRQLkLuLDjJd7u8wJs/mRs22J7CR4re3Qq3xIpENPUUHG+biuqX2UVsVX41epTp9ZjINm0ttqTIEZECQboK0ztd/Ya951t+nRi2GFYgksxluSlO7BMTYLVblMVcmOOTEZ9N+i4YvkREmAySz2JaNE4CqtM5jvrZHcbbE5lmLU8NrJTEmc3drsT1VlmmXdsCbgTdSu7wKUi2zPqsaq7GWZCjeJBZnwxlAN9dRyecBtsvzH/4+DV+MNdsLT4/GJSU0jHde3T2KtMxYnA8X5tvT5cp2eehDLnVxrPPoLgWpx4Hd+8Lp/Zx7boYoe/g40eDtViIF6Se+xPX8qPl96DqECImrRMOY2fg3e1atwd16UX1cR75wEXr4VmGdxL52atbpX2HnxfTenfiJcelKr9CjvQp/IMWJ57AO62ndOKk7890/iSWBU4XaZc0AAAAASUVORK5CYII=)","fist.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAABj0lEQVR4Xu2S3yuDYRTHz7IZsfdCUrYLG4VCpJV6d0mS/A1zg79ArRArP4q5s4ulVousFCn/gX/g5WK7wkZMzIXMxjZ793XeMdn7vpM/wDn1PD1P53PO95znIfq3rwmYyU5tZPr7PJYoTjGaZ8xB1t+xZQqyP60iAEeWsThFGKpqbiE9CRECrnCHLnTAj17QbLV4E3mFXALbcOAGSXRiClnMgZLkZR/QYg0UEpDBDuxI4IHzT6OADdRiEK2gE12gBWmWE8Yj7weQIGOT650zaoiqAVtjDx2KeEWR8xZ4lUt+jEWkWJxJBYgkWa7NLy5WXWl5FilzhfoKwEAzzkwYI3DhTQUoVYpYR10FYCTPaDbJ2heQUwGfR0kFELmt6Utu9Jmz6dmFpmmR7v141w1XBiCBNFNacyKmCxT59kgHGBIQ4g7UkpRzESuwaCoQ+YZxqwGUGaUwrvvStqazLeRLj/bTZUTQB5rQfg1jjac9e8qAzFgZKbDMALpj1Kz3Z0WKjiGIfex++x6H98PgK4d/APiDpdwStyBoAAAAAElFTkSuQmCC)","hand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB50lEQVR4Xq2TwUsUcRzF3yy52TboHiKVLBCElb1479g5kgKPCkKs4ME6eAiiP0GELileStew6NLJq5Sihw6BG2VlECymB2PVXd10Zl7vN+OuOjNLl35fZhiY+fzee9/vb4D/tAow9fafu90BsqrvnXzNBwSTTBAZ2I3BkSbeYBMzLHNRn9/nEFHCvYZAutDHDfb5wDvanOVnoozBhkBzIUeHOQEHfO8DW0QV0xjC3VjoUh2onAC/ZPAme4giemMQo+BKoaeusCmdSa41MmYL8HxLgUKe27q/UKLzSSxcwUOMwk7VgTKX6sBz7oeA61hJHSUP8ST1raZQA7Z8hb0QkO1WE+d5lZdlx+UbTsjSR95i0bc0E7GUzfIHdziouRqFPzwU5ii4IzOP+SFiqe3i8jyP1A1bgKPyVGa5/M2S7iaDdWZ8KUwNe1Uus923FHx8ujwZPA9YGO48+CK/Ywp4un8N8SIZgG78fCTvFV1n9w+ePe6GFMzMJ7u4zuOInQDYYaKcDB3Ba63FGQWPLk8mt2MAWKv96kk0sKM2TBFfkQ4fvkx6fyHSH9NalwO0PkXPaktirt8rhRpqelblbV6IASzkmivP9NrEdE9AM/FXbCPG4v+6p+3VcZ2ql5zTZSqv+XcQ6+gKgL8a/RiIJPdySgAAAABJRU5ErkJggg==)","handPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACAUlEQVR4Xq2Tz0sUcRjGn1lytW2wPUQamSAIK3vp7kk6i1LgMUGIFTqUBw9BiPsPhNAlpUtuKhpdOnVUKqlDh6CNsh+CsJQdjC13ddOZeXq+M7uTzu7SpXmZLwvzfr7P8/5Y4D89eZh48s/bhoC04nMXH/EmwThjRAp2c/B6C7vZwhRLXFP6NY4RRVxpCiTzw/zCYR94RpsP+Z4oYbQp0JbP0GFGwB6f+8A2UcF9jOFyQ+hkCJSrwDcZ7GcfUcDFBohRcKXQFyp8lc4s3zYzZgvwfEuBwgK/65xXRccrsXAGE7gBOxECJb4IgQfcjQAX8DJxEN/HVOJTTaEGbPsKvyJAuldNXOZZnpIdl485I0tveIkF31KuzlI6zU3ucFRzNQq/uS/MUeGOzNzm6zpLHa3ryzxQN2wBjsIT4JE6f7Co09RgHRlfAnPjXoXr7PQtmdSsNqn2eDJ4HLAw3rX3QX4nVaC534sA0bYCvdi6Je9lveb+mkLw2+PPiIKZ+WwPN3ioj9MyczQMsMNYKR5ZwfOnCzkVTq7KTpYDgrKCp5XuyGw9AOvViHoSmPhryaS7nCM+IhldvlRy92nYmaBLQWtdXqX1rn5X22OLI16xihhjBjA9q3CQJxoAFjJt5Xv6bBLdKmgmvsIOYrLxv+5uZ+WOtmqJi3pNLGj+54gN9ATAH0reBFwT+FQbAAAAAElFTkSuQmCC)","invisible.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACjUlEQVR4XnWTS09TURSF178wMUExvhIfiY+0Q6MThwwEHYk6MODAqHHgzD9gHDgWYhQBAUWMqY/SogSkCGJJNAiFUiyV2qqURkFa7cPvnJZiBXMH7Tl3rbXX2ntf5bWgyH+emPI8v8reAn8nj3r0Qr3qU78GNKghDWtEfk3ot2LKKq6AphTUjGYF26ObuqU7alabOvRQj/VEz6APKS23jvI/rY/6AD2ksCH0AG9Sq9r1QJ0W/hyREFbcOqQNcqiL/3FuIpozhJeot6J+H/VHegrci60lSA7t0VVtlRNqTvPAY4bQqxYL70LdxSsvd2l164B2qVHLVN4L0U2VhL4aQl9RveDdS9gl4IcBNWpRUShNUJxUT3OG0E/UTo4u4N0a1U9SHdRu3QOQIOw8N+0Yc/A2ZwgDeHfh3ZgZpbBH+zDTAPy7Jok6qU/K0Jht2k91CL5SVD9aHh1BvYHiCxYe1nV9w9h5bVcdk4DwGn3TmYg1Y9Rb8P2FQYVQbOM+qjPaDDwMguMw3j00Mst0narUDaUgBICPI5VgAhe5rUVguZBhhMaGOEzrrU6pQpfpdoqeB8k2hvoF7UB9mhxmsyD4WYIs0c/ifEKnoVxSEsptQke5rdQ5vGdRt4SYXTEf2lWq1l1s1GojlDiAgpmTLN4KHEKegw+NKvsco8oglApd0XtoOzEzhZmCuq2QJ7RRX3lqmO84xjaxGltUj/dVcJEwhuIqwVRptsZMZwL/wG2FPIHryijVVBnUNdJlS1ZKlmJEzkApr1JDj5KsyeJaQsR+fDlW8G/KCWyl1tgpziEIxczwVcnYcT6p5LpwO7gZ5jzH7mT0hmbW0+IO/UBixXX5L4RZlmpOn8u6vT7Y3P4BKLGmkBK1qvgAAAAASUVORK5CYII=)","lasso.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADg0lEQVR4Xk2UXWxTZRjHn7XYfWhh4DKlIFLRdlucImbGr2gEwYwsAjGRXXihRlTOFrOoF0RcdWwhEVnQdgOmbgzoliEhAZJlvZwdMQg6No/U2IYOWVrK1q1d19q1Pef8fU7bsy3/nIvznv/ved/n47xEZKDivNpK8RoE6XC4J9gX7A/2hXukwxCwta1UcxhUu4PcLFHnqZnaf+/E0FhTqDpiiaqqjjSFhsbvnZja76kRdarLQRzbTSjAuozgczaHVoJYhShX1krlSmH2bSWaQz5nRsA6MJQDrHMtx0Y3y4QS7Eo0eTpG3K5rP7tdHSNNnl2JNQxtlo+NzrXAmgXEDcnWA/5ihbAtOeyKtKf3wmIvs6+2l8GS3hv51u3aliQUKwf8yVZxA2GNIvSIBCNaJmPt/jrR6KZfyL6omQdQF2tvmTTyPt2iIhDq/f1mth/1LtjCVTa9QAIN8dEl8tK8mgJBj6oF21GvEWb4+wlft08Sds+FHbPmDl2A/mOl2Ran7SQylkV0MIcdu+cI7J3utUb0aL2q1DZSlDiRvObpKdpCZymVf1dqW6/qYY1QsM8SXYULw3jcq8XLWlSghOrJpYUwXxpeDUuUZn6smNWheUx6l7uxGD8HrKXz9A2doQidJnnPoWt6VEQItm4/4aX0LSd2wLCEqMB6+otmqYt8K1LVd3+oTRA6/yXUTXe+mCHsjE048QbWw8R6EHoNAMlGbA+d/PSOAZXKZC9tKbr19m+DFZztc/E/B2/3jp//x5logyC/8oyBd9DBIjeODj4/X8hTcO4PZZ/a6RKpYfBKWb6CuTQ+i41elD5A5cLHga7jPjOvWDIDv6e/EE25WXo49N0mPmF21OR3wm/FTahMf3/dNtLoe4GPa1LeC9y4xGUxacNnvHnykfTG+IWxJzOlyhnPzHH7bUJRvgINiSuu+CE8i2Jw8NwORq/jidRDktMz4COU46twnVyEx+brwxfFQDc+weu4P0drQFHq/Z+uP73UZq7vh8FAV6oeL2OTZl4OFMCUaZw4NXDjyNTBWENylaLDwXFpH+5Tm5mku4sN1XZQFwrkR+OvyjvxpvT5kTuEHcmpDh5i/qIQ/1l5LQd4RifUOS1A1XTnRq76r5dhzRlTPBxx7vlSDstmiA0r8NGXUcLZm6jJARkGZvnJAtqtod4JmhJbT/9tiZ4bS9QsX83dGv8DIrFVpnd+f4UAAAAASUVORK5CYII=)","lassoPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADiklEQVR4XlVUbWxTVRh+12K3oYWBy9SCSAHbbXGRaGb8SoygmJFFICayH/5AIyrdYhb1BxFXHWswQRa03WVM3RzYjaEhYcZl/Tk6QhB0bN62xjV0yNJStm7tutbetvfex3N7e3XmyUnuOXme9+O8zz1EZKDyIhwVeBE28VisLzIQGYwMxPrEY7Bhh6NCYxgUuou8DLwuUD936G736GRrtC5uSSioi7dGR6fuds8dCtTzOoXlIhbbSyjBhrwt6G6LrgExlKJKfkiskksLuzVoiwbdeRs2gIlUgXWp/eTEdomwGnvSrYGuca/n2g9eT9d4a2BPej0TbZdOTiy1w1oQ8JsyHYdD5TJhZ2bME+/M7YfFWelc56yEJbc//oXXszNDKJcPhzId/CbCetnWxxOMaJ9NdoYaeaOXLpHg8HOCw0lOWrgPjcnO9lkjy9PLyzZCU2jQzOgnpgV7rNaut5GNRsnPEXzcstICQY9awX5i2ggzQoOEzzpnCXuXYq5Fc5cuTH8z5IoCnkRVooM55tq7RGDc+X5rXI+Oq3JDCyVILhTj54bPEYbPKV+CoyAhuaHjqh7WOEUGLIm1uDCGbdOFeEoxK+Hjilvz8Ng6WBK08E31og5tk+IBNg3GFBw+zlfMoHwJjrMUpzMk7Tt6TY/qOMHeGyI8l7vpxi4YtOBq0xvJR4vUQ8FV2bo7XzekCdxfhMZ57tk8YXdyxo1XsBEmhvv93ZoAJBnxcvT0B7cNqJFn++mJspuv/zJSzcb2VOr3kVv9Uz/+6U47BI/vZ+FzlkEHi9QyMfL0cilzwfnf5IPKpFeLzSOXK4s3qDb8YXLiovg2aoT3wj2ngmZ2YskP/Zr7mDepXnow+uVWVmHBatIbsddSJtTkvrpuH28JPsPKNclvhm8Ms2sxaeYz+k8/nNucujD5WL5CPhtYOOW8RSgr3kBz+rIndRRPohwsuJrBOO16NPuA6A4MBQlV+DTWKJVhy3JT7CIf7sX7eAn3qmpNUJZ969vrj7PGNRjwTiTck23C89iqkVcKSmDKt8x8N3Tj+NyRZHNmrazDkSnxIO5RhpmhO/9G0jIoByXSI6kXpN14Vfzo+G3CrsxcFzOx4iJif1YRKwXMozNssZi189xmdutXfoJVJWaZOVJs5v/18H+/rcK7nyQI3/tRrwryTLDIVkGgvRrKm6AhvePMH5bE+cl0/cpT9dX4B08ibwJFFp9KAAAAAElFTkSuQmCC)","pointingHand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAADe0lEQVR4Xq2V+yvrYRzHnWwuaaUst9mY21iyHcVySghxJJLLDy7L2Y9S4geHThx+5ISc0vCDf0AS6fj1rPxkRJ2dJjvtjNolt1y+NMY+5/N5yonZbIunvm179nyf1/N+fy5PSMgLo6OjwxAWFgaxsbHQ2NhonJiYELy0Pqj/CgoKPqSmpjoGBgZgfHwciouLna2tre+D2uSlxXK5XJ2UlMTt7e2B2WyGjIwMiImJ+fhmADy9OjMzk7PZbHB0dAT4HXp6en6+GSA5OVktk8k4u90Ox8fHTEF3d/fvNwNIpVKm4AFACgYHB//Mzc0pUZ3o1SBPgEKhgKysLFd5ebkdlekTExMLUKUU56VVVVVhQQM9AdnZ2dDe3g5bW1tQU1MDKSkpNrTtr0qlsvT3939/NSA9PR2mp6fh7u4OhoaGWG2gXdDZ2Qnx8fFWBH3CR4P1IvMFe4feqoqKir7k5OQoSQEF2Wq1siwiwMzMDLjdbhgeHoa8vDw4OztjiiIiIqCyshLq6uoA39mRSCSKZ5DQ0FAxnuQXeutG+Yb8/Hwt1gIDHB4eMsDs7Czc399TsAFPCxzHwe7uLojFYlhdXQWDwQBKpZLDfdTeVMhra2vBaDRCV1cX2zAtLQ0IcHp6yuYWFxeZRfPz89DX18cAVIgYbNje3oaTkxPAJODwsN4BLS0tTPbGxgagTBCJRECFdnt7C+fn53B9fc0UXF1dweXlJbhcLnA4HAxmsVhYvVRUVHA8Hu85IDIyUl5SUgImk4mduKmp6T+ANiXvHz+Ag+ZJEYHpkxTgHt4VCIVCCTY009TUFNzc3MDk5CTlPRwcHLCN/A2CE6C0tNQ7IDw8nI8Z8LWhoeGWFGxubkJvby+THSyAz+d7jUEIpp46ISGB0+l0zHeKh9PpDAhACimd0QUuKirKO4BSC3uOTqPRwMXFBQvig/+BWER9C+8RLjo62jcAPayhvqPX6xmAgkf++hu0huKF2cdhivsGjIyMCAsLC81jY2PMHgIEMgiwvr4OcXFxHNaCbwDZ1NbW9q2srAz29/fZ6QOB0LrR0VHAdPd/ZwgEAhF6uaPVap/EwbMOHpTRPGUbKgfsBv5vPawJXn19/WfsR04qPMoozzg8hlEirK2tMUBzc3Pg93Zubq6uurqa9aCVlZUnz/LyMvu9tLQECwsLgPVD1f/Ds8n9AyubuM4FiwqZAAAAAElFTkSuQmCC)","rect.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACc0lEQVR4Xo2UT0hUURTGf7uipMy8GbUwigx0WbiwpCgQKmgl1CYj3Plahtg/eguFUIoamNxUCEFCGlIgEhSORmQRNeVMSkhpWcSU1KSOQ9i8vvfmpiMZOB8Xzjv3fPPd8+dertPHQICX/OAVnTRTz1GOaNXL7pQvSdTG9KHwCVJMyzlMFw1UsJFC8lmtVSi7Qr4u7SUVk1KsmCkyZtJETchUmTKh3NSYk6ZBq0a276nSXtRMGc+kfMIkiRY3shQkWiZ9wjjxMN5SEAuPQ5qegBALe07a+eiMCj9lxZ0ZJ+N483A9oQf6qbWED6rITW4IcQGaGJuT/Y0f7rm1cJliS+jnLjNkAviEDXTwVbZ/2gwjobLESKgY6iiwhI5gyw/N/T2zKtljF0CN6p39SAZbCVyhTpwVLKOUM8wGmn4lEy35yGH+Jp1TKF9nrRp3mzbOMcp9m0MhtLPrvwRDr/6/mwda6cZ4ON24Hd5z1RISfGNIAdmDnWabtHtlzxLjtY1pgucM2o8rPOUhj+UZY0qDWGEJad5y0ca8mO90LLyKA+yjkirN6QWeaO0kwjvucZw8m0MwGinLzmelUi1nP3vYQrUCH3GNY5Qp+UpbpelcQjNnWcchVWRQhc2jiPWsYTl7aZRetvTBtP5VaJW7la0c5g4npFbCbk5xSzlNKPGRUKk6vYCwScIH2ax2lagH1VJ6wxdbNVUumKUFhGFNaMxiyEk4v3Jn1VmE4LluBK804W/5zdDA/+OxCtk58dzeNjcS7vYJ/v1bzGMJnm7rPDLmkzlvdph2E5/zRs13M2BS/p3OvhoLkVSBi7jE5zn/VPCy+K/GH86K+gV5WsL6AAAAAElFTkSuQmCC)","rectPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACi0lEQVR4Xo2UT0hUQRzHP6c2KmrTnYoKjCAP+45FB0uKAqECT0FdMsKbr2OIUdE76EUpamHzUmoEGWmIgUnQn9UOWURp7ksJKSuLeCW1qesStq/fvB11lzr4vgzM/Ob3md+b3/xmuEYfA4Fe8ZMhOmmklmMclVYr/U6xpRg0Pn2I+yRpZsQ4Shd1lLGJCGHWSItIv0xsXTKXEp+0+AqZJqum1KCKqQpliXaqKnVK1Umrkr62VMjcoJpWvkprYAqvyUksRV7TlAYmcOP4S1EyPgEZegMgGfftjP3JHhf9kp5rz9pZ21+U44t6oZ9qA3yUjNygReSKoIEPC2H/oN19pxouUWKAfrqZJRtIAxvp4Jv09d9mGYtZ3lisBGooMkBHMKVd5QsRlSbfcxMl99tFUCX5zg1SwZSHI6qJ4q+wNHWGuSCmzqTXFEYMan7TeYlyBSi2ItymjXOMc9/sIQLt7C4AMiE36ka7K/HDlUXR1uh0qIcH9JCpd+OZ+u3wnisG8PjOCEOydr6S1hxJXhufBnjBsBlc5hkPeRS6bt2zbuUiWK1WJpThLReMz8vFk07GV3OQ/ZRTIXWqN11t7SLBO+5yglVmD0FppA0dZqWkcScH2MtmAVqsJ1zlOJZUbbnJ0kw+0MhZ1lEpGRnGCS2zVGgDa1nOPup5ahYNqnU+QrOYm9nGEe5wUqKVsofT3JRdTspZjMWictIFwBYJfIitcsClFHNYIr3hq7iapAW1VACMSoUmjUZsz/6dX6t2rvgKAN9xEvhRT0/oZaXg/7EYIFcnvvO4zUnEezSg79//LAbw5bYuKqs+q/Nqh2pX7oJ1UP1QAyqt73Tu1ShUShK8not8WbBPBy+LfjX+AjC0Avv8MQHTAAAAAElFTkSuQmCC)","zoom.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACiUlEQVR4XpVTTUgUYRh+nJWCxUsNevBiYGm7UjAnFyoI9ORKSxt4a71Ei9I10EPg0VO7hOE1ikQSofXiYv6QRX9bkTAjbbM/B2mz8id3Z2f/ZvbtndR0NYn9XuYbPuZ55n3f730efMQCZjCLOX4/xyJe4BXe4C3e4QNi0JDCBu8JRPkURxIMm8BjjOMJJvEUU8jadUn36D79Zs5X8JhSyb6AX/iBzxxfLMIMwx/gIcYwXRMTNa86Mhrp23QXOwx3sW9zNKKOaF5NzAlrnCWBFXAx43jE8PeC0b4U9KstJmh/tJh+dSlotJeEDYZ/Bea5GAtudoZDkm5jcCP1poejd+XhaG+6kc82kvRwiDpLwha+A1RX8K4OaDfCU038UTQGk/GA2qO4lDbFpfbEA4NJ0QA1UThktOeFDBPOLd+/qF9PtOZBDi30LNU91FBfgz+rvmaoIdUVmnZo4CxLQU3cYsLJ7G1HebvqC7nIhHJ6G7y7mmsNKTQtGjbyq5p3Dnzv2au31i24QK35cEhpriQAzbUp92AS1GKqI7odL5F1jM9bzV7KzoyVLvfbDhKAoYZ4wGp/NFKU8Br6scS9VrM7Fw8a56/8A271ovb0pkF9m2UPy2AWGd/k4sYdpal/7+fH4cTxvaPsGo6C3EXysWYWQafIR2cqSnFyV237CM6ADOowyM8Si+xkkCsztB2RIYZCFT2QB1mYVdwSSSza/N85nD1qDl27cyA76EQ1k6bqtESiRagrXFsdyBxQq+ySnfIhtbJ6gHX2bBprQvmAHwKH/GDBmRCDyibP4Od/HLcNZ0KcCSnkmLQsKEd4mmvfgTMhybGCb0zI4RPb1bSbUtnDUvHz4yGJ7BUWx2+5Iu7Rc33AhQAAAABJRU5ErkJggg==)"};var q8=Object.create(null),gu=3/4,JP=class{constructor(ft){this.HOTSPOT_32=ft.HOTSPOT_32,this.IMAGE_NAME_32=ft.IMAGE_NAME_32,this.cPointerCursor=ft.cPointerCursor,this.cTextCursor=ft.cTextCursor;}getCursor(ft){if(q8[ft])return q8[ft];if(this.IMAGE_NAME_32[ft])return this.buildCursor(ft);switch(ft){case this.cPointerCursor:return "default";case this.cTextCursor:return "text";default:throw new Error(`Unknown cursor: ${ft}`);}}buildCursor(ft){let vt=this.IMAGE_NAME_32[ft],Ot=`${wu[vt]} ${this.HOTSPOT_32[ft*2]*gu} ${this.HOTSPOT_32[ft*2+1]*gu}, default`;return q8[ft]=Ot,Ot;}};var ZP=class{constructor(ft,vt){this.title=ft,this.rootElement=vt,this.elements=[],this.dialogElement=null;}setLayout(ft,vt){this.hLayout=ju(ft),this.vLayout=ju(vt);}add(ft,vt,kt,Ot,p1){this.elements.push({component:ft,x:vt,y:kt,x2:Ot,y2:p1});}createTextField(ft,vt){return new S8(ft,vt);}createLabel(ft){return new E8(ft);}createComboBox(){return new V8();}createCheckBox(ft){return new x8(ft);}setEventConsumer(ft){this.consumer=ft;}showMessage(ft){window.alert(ft);}showDialog(){let ft=document.createElement("dialog"),vt=this.rootElement.getBoundingClientRect();Object.assign(ft.style,{position:"absolute",marginBlock:0,left:`${vt.left}px`,right:`${document.body.parentElement.clientWidth-vt.right}px`,top:`${this.rootElement.offsetTop+30}px`}),this.dialogElement=ft,this.rootElement.getRootNode().append(ft);let kt=document.createElement("div");kt.style.display="grid",kt.style.gridTemplateColumns=this.hLayout,kt.style.gridTemplateRows=this.vLayout,ft.append(kt);for(let{component:t1,x:at,y:g1,x2:T1,y2:k1}of this.elements){let i1=document.createElement("div");T1===void 0?(i1.style.gridColumn=`${at+1} / ${at+2}`,i1.style.gridRow=`${g1+1} / ${g1+2}`):(i1.style.gridColumn=`${at+1} / ${T1+2}`,i1.style.gridRow=`${g1+1} / ${k1+2}`),i1.append(t1.getElement()),kt.append(i1);}let Ot=document.createElement("div");Ot.style.display="flex",Ot.style.flexDirection="row-reverse",Ot.style.gap="15px";let p1=document.createElement("button");p1.textContent="OK",p1.addEventListener("click",()=>{this.consumer.fireOk();}),Ot.append(p1);let zt=document.createElement("button");zt.textContent="Cancel",zt.addEventListener("click",()=>{this.consumer.fireCancel();}),Ot.append(zt),ft.append(Ot),ft.showModal(),ft.addEventListener("cancel",()=>{this.consumer.fireCancel();});}disposeDialog(){this.dialogElement!==null&&(this.dialogElement.remove(),this.dialogElement=null);}},$e=class{setEventHandler(ft){this.eventHandler=ft;}fireEvent(ft,vt){this.eventHandler(ft,vt);}},E8=class extends $e{constructor(ft){super(),this.element=document.createElement("label"),this.setText(ft);}setText(ft){this.element.textContent=ft;}getElement(){return this.element;}},S8=class extends $e{constructor(){super(),this.element=document.createElement("input"),this.element.type="text";}setText(ft){this.element.value=ft;}getText(){return this.element.value;}getElement(){return this.element;}},V8=class extends $e{constructor(){super(),this.element=document.createElement("select"),this.element.addEventListener("change",()=>{this.fireEvent(2,this.element.selectedIndex);});}setEnabled(ft){this.element.disabled=!ft;}addItem(ft){let vt=document.createElement("option");vt.textContent=ft,this.element.append(vt);}getSelectedIndex(){return this.element.selectedIndex;}setSelectedIndex(ft){this.element.selectedIndex=ft;}setSelectedItem(ft){let vt=this.element.options;for(let kt=0;kt<vt.length;kt++)vt[kt].textContent===ft&&(this.element.selectedIndex=kt);}getSelectedItem(){return this.element.options[this.element.selectedIndex].textContent;}removeAllItems(){this.element.innerHTML="";}getElement(){return this.element;}},x8=class extends $e{constructor(ft){super();let vt=document.createElement("label"),kt=document.createElement("input");kt.type="checkbox",kt.addEventListener("change",()=>{this.fireEvent(3,kt.checked?1:0);}),vt.append(kt),vt.append(ft),this.element=vt,this.checkBox=kt;}setEnabled(ft){this.checkBox.disabled=!ft;}isSelected(){return this.checkBox.checked;}setSelected(ft){this.checkBox.checked=ft;}getElement(){return this.element;}};function ju(d){return d.map(ft=>ft>0?`${ft}px`:"auto").join(" ");}var Sn=class{constructor(ft){this.imageData=ft,this.dataView=new DataView(ft.data.buffer);}getWidth(){return this.imageData.width;}getHeight(){return this.imageData.height;}getRGB(ft,vt){let kt=this.dataView.getInt32((vt*this.imageData.width+ft)*4,false);return (kt&255)<<24|kt>>>8;}setRGB(ft,vt,kt){let Ot=kt>>>24&255,p1=kt<<8|Ot;this.dataView.setInt32((vt*this.imageData.width+ft)*4,p1,false);}toDataURL(){let ft=document.createElement("canvas"),vt=ft.getContext("2d");return ft.width=this.imageData.width,ft.height=this.imageData.height,vt.putImageData(this.imageData,0,0),ft.toDataURL("image/png");}};var KP=class{constructor(ft,vt,kt){this.canvasElement=ft,this.dialogRoot=vt,this.JavaEditorArea=kt;}register(ft){this.javaUiHelper=ft,this.cursorManager=new JP(this.JavaEditorArea,ft);}grabFocus(){this.canvasElement.focus({preventScroll:true});}setCursor(ft){this.canvasElement.style.cursor=this.cursorManager.getCursor(ft);}showHelpDialog(){}createImage(ft,vt){let kt=new ImageData(ft,vt);return new Sn(kt);}createImageFromBase64(ft,vt,kt){kt=kt.replaceAll(/%\d+%/g,t1=>"A".repeat(Number(t1.slice(1,-1))));let Ot=Tu(kt),p1=new Uint8ClampedArray(Ot),zt=new ImageData(p1,ft,vt);return new Sn(zt);}createDialog(ft){return new ZP(ft,this.dialogRoot);}runLater(ft){typeof requestAnimationFrame=="function"?requestAnimationFrame(ft):typeof setImmediate=="function"?setImmediate(ft):setTimeout(ft,0);}};function Bu(d,ft,vt,kt,Ot,p1,zt,t1){let{readOnly:at=false,initialMode:g1="molecule",initialFragment:T1=false}=ft,k1=document.createElement("div");k1.dataset.openchemlibCanvasEditor="true",Object.assign(k1.style,{width:"100%",height:"100%",display:"flex",flexDirection:"row",alignItems:"start",backgroundColor:"white",touchAction:"none",userSelect:"none",webkitUserSelect:"none"});let i1=k1.attachShadow({mode:"open"});i1.adoptedStyleSheets=[bu()];let o0=null;at||(o0=document.createElement("canvas"),i1.append(o0));let u0=document.createElement("div");Object.assign(u0.style,{width:"100%",height:"100%"}),i1.append(u0);let W1=document.createElement("canvas");W1.tabIndex=0,Object.assign(W1.style,{outline:"none"}),u0.append(W1),d.append(k1);let q0=new p1(new KP(W1,u0,kt)),St=new kt(u2(g1,kt),new OP(W1,vt),q0);if(T1)if(g1==="molecule"){let n1=new zt(0,0);n1.setFragment(true),St.setMolecule(n1);}else {let n1=t1.create();n1.setFragment(true),St.setReaction(n1);}q0.setEditorArea(St);let qA=at?null:new Ot(St,new GP(o0),q0);function rA(n1){W1.style.width=`${n1.width}px`,W1.width=Math.floor(n1.width*_A),W1.style.height=`${n1.height}px`,W1.height=Math.floor(n1.height*_A),n1.width>0&&n1.height>0&&St.repaint();}let Vn=u0.getBoundingClientRect();rA(Vn);let C1=new ResizeObserver(([n1])=>{rA(n1.contentRect);});C1.observe(u0);let m0=null,ve=null,st=null;at||(m0=H8(W1,St,kt),ve=cu(u0,W1,St,kt,zt),st=H8(o0,qA,kt));function q1(){k1.remove(),C1.disconnect(),m0?.(),ve?.(),st?.();}return {editorArea:St,toolbar:qA,uiHelper:q0,destroy:q1};}function u2(d,ft){switch(d){case "molecule":return 0;case "reaction":return ft.MODE_REACTION|ft.MODE_MULTIPLE_FRAGMENTS;default:throw new Error(`Invalid initial mode: ${d}`);}}function du(d,ft,vt,kt,Ot){class p1{#t;#A;#P;#n;#f;#r;constructor(t1,at={}){let{editorArea:g1,toolbar:T1,uiHelper:k1,destroy:i1}=Bu(t1,at,o0=>this.#i(o0),d,ft,vt,kt,Ot);this.#t=g1,this.#A=T1,this.#P=k1,this.#n=null,this.#f={[d.EDITOR_EVENT_MOLECULE_CHANGED]:"molecule",[d.EDITOR_EVENT_SELECTION_CHANGED]:"selection",[d.EDITOR_EVENT_HIGHLIGHT_ATOM_CHANGED]:"highlight-atom",[d.EDITOR_EVENT_HIGHLIGHT_BOND_CHANGED]:"highlight-bond"},this.#r=i1;}getMode(){return this.#e(),this.#t.getMode()&d.MODE_REACTION!==0?"reaction":"molecule";}setMolecule(t1){this.#e(),this.#t.setMolecule(t1);}getMolecule(){return this.#e(),this.#t.getMolecule();}setReaction(t1){this.#e(),this.#t.setReaction(t1);}getReaction(){return this.#e(),this.#t.getReaction();}setOnChangeListener(t1){this.#e(),this.#n=t1;}removeOnChangeListener(){this.#e(),this.#n=null;}clearAll(){this.#e(),this.#t.clearAll();}destroy(){this.#e(),this.#r(),this.#t=null,this.#A=null,this.#P=null,this.#n=null,this.#r=null;}get isDestroyed(){return !this.#t;}moleculeChanged(){this.#e(),this.#t.moleculeChanged();}#e(){if(this.isDestroyed)throw new Error("CanvasEditor has been destroyed");}#i(t1){if(!this.#n)return;let{what:at,isUserEvent:g1}=t1;this.#n({type:this.#f[at],isUserEvent:g1});}}return p1;}function mu(d,ft,vt,kt){class Ot extends HTMLElement{static MODE=Object.freeze(Object.create({MOLECULE:"molecule",REACTION:"reaction"}));static observedAttributes=Object.freeze(["idcode","fragment","mode","readonly"]);#t={idcode:"",fragment:false,mode:Ot.MODE_MOLECULE,readonly:false};get idcode(){return this.#t.idcode;}set idcode(zt){this.#t.idcode=String(zt),this.setAttribute("idcode",this.#t.idcode);}get fragment(){return this.#t.fragment;}set fragment(zt){this.#t.fragment=!!zt,this.#t.fragment?this.setAttribute("fragment",""):this.removeAttribute("fragment");}get mode(){return this.#t.mode;}set mode(zt){this.#t.mode=String(zt),this.setAttribute("mode",this.#t.mode);}get readonly(){return this.#t.readonly;}set readonly(zt){this.#t.readonly=!!zt,this.#t.readonly?this.setAttribute("readonly",""):this.removeAttribute("readonly");}setMolecule(zt){this.fragment=zt.isFragment(),this.idcode=`${zt.getIDCode()} ${zt.getIDCoordinates()}`,this.#A.setMolecule(zt);}getMolecule(){return this.#A.getMolecule();}setReaction(zt){this.fragment=zt.isFragment(),this.idcode=kt.encode(zt,{keepAbsoluteCoordinates:true,mode:kt.INCLUDE_MAPPING|kt.INCLUDE_COORDS|kt.RETAIN_REACTANT_AND_PRODUCT_ORDER})??"",this.#A.setReaction(zt);}getReaction(){return this.#A.getReaction();}clearAll(){this.#A.clearAll(),this.idcode="";}moleculeChanged(){this.#A.moleculeChanged();}#A;#P(){this.#A||(this.#A=new d(this,{readOnly:this.readonly,initialMode:this.mode}),this.#A.setOnChangeListener(this.#l),requestIdleCallback(()=>this.#n()));}#n(){switch(this.mode){case Ot.MODE.MOLECULE:return this.#r();case Ot.MODE.REACTION:return this.#e();default:throw new Error(`Mode ${this.mode} is not supported`);}}#f(zt){let t1=zt.indexOf(" ");if(t1===-1)return ft.fromIDCode(zt);let at=zt.slice(0,t1),g1=zt.slice(t1+1);return ft.fromIDCode(at,g1);}#r(){let zt=this.#f(this.idcode);zt.setFragment(this.fragment),this.#A.setMolecule(zt);}#e(){let zt=kt.decode(this.idcode,{ensureCoordinates:true})??vt.create();zt.setFragment(this.fragment),this.#A.setReaction(zt);}#i=false;#o(zt){this.#i=true;try{zt();}finally{this.#i=false;}}#l=zt=>{let t1=this.idcode,at=this.fragment;this.#o(()=>{if(zt.type==="molecule")switch(this.mode){case Ot.MODE.MOLECULE:{let T1=this.getMolecule();this.idcode=`${T1.getIDCode()} ${T1.getIDCoordinates()}`,this.fragment=T1.isFragment();break;}case Ot.MODE.REACTION:{let T1=this.getReaction();this.idcode=kt.encode(T1,{keepAbsoluteCoordinates:true,mode:kt.INCLUDE_MAPPING|kt.INCLUDE_COORDS|kt.RETAIN_REACTANT_AND_PRODUCT_ORDER}),this.fragment=T1.isFragment();break;}default:throw new Error(`Unsupported mode ${this.mode}`);}});let g1=new CustomEvent("change",{detail:zt,bubbles:true});if(this.dispatchEvent(g1),zt.mode==="molecule"){if(this.idcode!==t1){let T1=new CustomEvent("idcode-changed",{detail:this.idcode,bubbles:true});this.dispatchEvent(T1);}if(this.fragment!==at){let T1=new CustomEvent("fragment-changed",{detail:this.fragment,bubbles:true});this.dispatchEvent(T1);}}};#D(){this.#A&&(this.#A.destroy(),this.#A=void 0);}#u(){this.#D(),this.#P();}connectedCallback(){this.#t={idcode:this.getAttribute("idcode")||"",fragment:this.hasAttribute("fragment"),mode:this.getAttribute("mode")||Ot.MODE.MOLECULE,readonly:this.hasAttribute("readonly")},this.#P();}disconnectedCallback(){this.#D();}adoptedCallback(){this.connectedCallback();}attributeChangedCallback(zt,t1,at){if(!this.#A||this.#i)return;(()=>{switch(zt){case "idcode":return this.#t.idcode=String(at),()=>this.#n();case "fragment":return this.#t.fragment=at!==null,()=>this.#n();case "mode":return this.#t.mode=String(at),()=>this.#u();case "readonly":return this.#t.readonly=at!==null,()=>this.#u();default:throw new Error("unsupported attribute change");}})()();}}return Ot;}function z8(d){let{GenericEditorArea:ft,GenericEditorToolbar:vt,GenericUIHelper:kt,Molecule:Ot,Reaction:p1,ReactionEncoder:zt}=d,t1=du(ft,vt,kt,Ot,p1);function at(){let g1=customElements.get("openchemlib-editor");if(g1)return g1;let T1=mu(t1,Ot,p1,zt);customElements.define("openchemlib-editor",T1);let k1=document.createElement("style");return k1.id="openchemlib-editor-default-style",k1.innerHTML=`
    /* dynamicaly added from openchemlib registerCustomElement with low priority */
    openchemlib-editor:defined {
      display: block;
      height: 400px;
      width: 600px;
    }
    `,document.head.prepend(k1),T1;}d.CanvasEditor=t1,d.registerCustomElement=at,delete d.GenericEditorArea,delete d.GenericEditorToolbar,delete d.GenericUIHelper;}function ku(d){d.prototype.getNextCustomAtomLabel=function(vt){let kt=vt||"1",Ot=new Set();for(let zt=0;zt<this.getAllAtoms();zt++){let t1=this.getAtomCustomLabel(zt);t1&&Ot.add(t1);}let p1=0;for(;Ot.has(kt)&&p1++<100;)kt=o2(kt);return kt;};}function o2(d){let ft=d.match(/(\d+)/);if(ft){let kt=Number.parseInt(ft[1],10);return d.replace(/(\d+)/,(kt+1).toString());}let vt=d.match(/([a-yA-Y])([^a-zA-Z]*)$/);if(vt){let kt=vt[1],Ot=String.fromCodePoint(kt.codePointAt(0)+1);return Ot==="Z"||Ot==="z"?"1":d.replace(/([a-yA-Y])([^a-zA-Z]*)$/,`${Ot}$2`);}return "1";}function YP(d,ft){switch(ft){case "superscript":for(let vt=0;vt<d.getAllAtoms();vt++){let kt=d.getAtomCustomLabel(vt);kt&&!kt.startsWith("]")&&d.setAtomCustomLabel(vt,`]${kt}`);}break;case "normal":for(let vt=0;vt<d.getAllAtoms();vt++){let kt=d.getAtomCustomLabel(vt);kt?.startsWith("]")&&d.setAtomCustomLabel(vt,kt.slice(1));}break;case "auto":for(let vt=0;vt<d.getAllAtoms();vt++){let kt=d.getAtomCustomLabel(vt);kt&&(d.getAtomLabel(vt)==="C"?kt.startsWith("]")&&d.setAtomCustomLabel(vt,kt.slice(1)):kt.startsWith("]")||d.setAtomCustomLabel(vt,`]${kt}`));}break;}}function Fu(d){let ft=d.fromMolfile;d.fromMolfile=function(kt,Ot={}){let{customLabelPosition:p1}=Ot,zt=ft.call(this,kt),t1=kt.includes(`\r
`)?`\r
`:`
`,at=kt.split(t1);if(at.length<4||!at[3].includes("V2000"))return zt;let g1=at.slice(4+zt.getAllAtoms()+zt.getAllBonds());for(let T1=0;T1<g1.length;T1++){let k1=g1[T1];if(k1.startsWith("A  ")){let i1=Number(k1.slice(3)),o0=g1[T1+1]?.trim();T1++,!Number.isNaN(i1)&&i1<=zt.getAllAtoms()&&o0&&!zt.getAtomCustomLabel(i1-1)&&zt.setAtomCustomLabel(i1-1,o0);}if(k1.startsWith("V  ")){let i1=k1.split(" ").filter(Boolean);if(i1.length>=3){let o0=Number(i1[1]),u0=i1.slice(2).join(" ");!Number.isNaN(o0)&&o0<=zt.getAllAtoms()&&!zt.getAtomCustomLabel(o0-1)&&zt.setAtomCustomLabel(o0-1,u0);}}if(k1.startsWith("M  ZZC")){let i1=Number(k1.slice(7,10).trim()),o0=k1.slice(10).trim();i1&&o0&&zt.setAtomCustomLabel(i1-1,o0);}}return YP(zt,p1),zt;};}var l2=["M  STY","M  SLB","M  SAL","M  SDT","M  SDD","M  SED"];function vu(d){let ft=d.prototype.toMolfile;d.prototype.toMolfile=function(kt={}){let Ot=this.getCompactCopy(),{includeCustomAtomLabelsAsALines:p1=false,includeCustomAtomLabelsAsVLines:zt=false,customLabelPosition:t1,removeCustomAtomLabels:at=false}=kt;YP(Ot,t1);let g1=ft.call(Ot);if(!p1&&!zt&&!at)return g1;let T1=g1.includes(`\r
`)?`\r
`:`
`,k1=g1.split(T1);if(at&&(k1=k1.filter(u0=>!l2.some(W1=>u0.startsWith(W1)))),k1.length<4||!k1[3].includes("V2000"))return g1;let i1=[];for(let u0=0;u0<Ot.getAllAtoms();u0++){let W1=Ot.getAtomCustomLabel(u0);if(W1){let q0=String(u0+1).padStart(3," ");p1&&i1.push(`A  ${q0}`,W1),zt&&i1.push(`V  ${q0} ${W1}`);}}let o0=k1.findIndex(u0=>u0.startsWith("M  END"));return o0===-1?g1:(k1.splice(o0,0,...i1),k1.join(T1));};}function pu(d){let{ConformerGenerator:ft,ForceFieldMMFF94:vt,Molecule:kt}=d;ft.prototype.molecules=function*(){let at;for(;(at=this.getNextConformerAsMolecule())!==null;)yield at;};let Ot={maxIts:4e3,gradTol:1e-4,funcTol:1e-6},p1=vt.prototype._minimise;delete vt.prototype._minimise,vt.prototype.minimise=function(at){return at={...Ot,...at},p1.call(this,at.maxIts,at.gradTol,at.funcTol);},Fu(kt),vu(kt),ku(kt);function zt(t1){if(!t1)return null;if(t1.includes("V2000")||t1.includes("V3000"))return kt.fromMolfile(t1);try{return kt.fromSmiles(t1);}catch{}try{return kt.fromIDCode(t1);}catch{}return null;}kt.fromText=function(at){let g1=zt(at);return g1&&g1.getAllAtoms()>0?g1:null;},kt.prototype.getOCL=function(){return d;};}function Fe(){}function Q2(d){let ft=d.document,vt={},kt={userAgent:"webkit"},Ot=Fe;Ot.__moduleStartupDone=Fe;var at="object",g1="anonymous",T1="fnStack",k1=`
`,i1={4:1,9:1,5:1,8:1},o0="Unknown",u0="boolean",W1="number",q0="string",St=2147483647,qA='For input string: "',rA="null",Vn="__noinit__",C1={4:1,5:1,8:1},m0={4:1,19:1,33:1,27:1},ve=" (copy)",st={4:1,9:1,14:1,5:1,11:1,8:1,13:1},q1=65536,n1=65535,K0=1e4,ae="fromIndex: ",N8=" > toIndex: ",XP=", toIndex: ",I8=", length: ",_P="Index: ",WP=", Size: ",Ft={15:1,4:1,5:1},ie={4:1},H={7:1,4:1,5:1},WA=16777215,Y0=65280,fe=.30000001192092896,EA=545460846592,QA=4294967296,SA=8589934592,VA=17179869184,xA=34359738368,zA=68719476736,NA=137438953472,De=274877906944,ht={12:1,4:1,5:1},r0=536870912,$P=2.617993878,aP=3.665191429,I1=6.283185307179586,A1=3.141592653589793,k0=1.5707963267948966,I0=16384,xn=-8355712,Z1=4096,tn={l:0,m:0,h:1},An={l:0,m:0,h:2},en={l:0,m:0,h:4},X0=1920,i0=1024,mA=234881024,nn=167772160,ue=100663296,oe=201326592,$A=114688,aA=4063232,Pn=3801088,pe=3538944,Re=3014656,rn=3145728,v0=2097152,te=393216,iA=917504,le=1966080,Qe=0xf8000000000,zn=0xf0000000000,Nn=0xe8000000000,In=0xd8000000000,On=0xb8000000000,tr=0xe0000000000,Ar=0xc0000000000,fn=8796093022208,Ce=549755813888,Gn=1649267441664,Jn=3848290697216,Dn=8246337208320,er=0xc8000000000,nr=9345848836096,Pr=0x98000000000,kA=29360128,IA=268435456,L0=8192,j1=2048,LA=32768,un=-1.5707963267948966,UA=130560,on=6291456,rr=-2.0943951023931953,b0=1.0471975511965976,Me=.5235987755982988,Q1=786432,yA=524288,sA=262144,O0=-16777216,fA=-65536,Zn=-32768,ir='<DrawingObject type="',Kn="></DrawingObject>",Yt={4:1,5:1,16:1},e0={4:1,5:1},fr="unsupported atomicNo:",O8=2.8415926535897933,E1={28:1,4:1,9:1,5:1,8:1},He={4:1,5:1,11:1,8:1},Y={10:1,4:1,5:1},c0={4:1,9:1,5:1,8:1,30:1},n0=131072,Yn=196608,Xn={l:6,m:0,h:4},hA=1048576,TA=3221225472,Dr={l:0,m:0,h:3},Le=8388608,ur={l:0,m:0,h:120},_n={36:1,43:1,38:1},s0={4:1,9:1,5:1,11:1,8:1},G8={36:1,43:1,95:1,38:1,83:1},Wn={4:1,5:1,36:1,43:1,95:1,20:1,38:1,83:1,126:1},U0=17976931348623157e292,or=-17976931348623157e292,OA={87:1,73:1,74:1,4:1,9:1,5:1,8:1},R1="??",bA={58:1,4:1,9:1,5:1,8:1},cA=2013265920,F0=32505856,E0=31744,$n=122880,qe={128:1,4:1,9:1,5:1,11:1,8:1},ln=33554432,S0=67108864,an=-131073,tP=-33554433,AP=-33292289,lr=-3.141592653589793,_1=4194303,V0=3072,w0={46:1,4:1,5:1},Qr=.7853981633974483,J8=3.061592653589793,x0=1073741824,Z8=541165879296,Vt={17:1,4:1,5:1,8:1},eP=281.17,nP=289.19,PP=294.21,rP="0.0000",iP="NOSEARCH_OCL_CUSTOM_LABEL",FA="M  END",Ee="ATOMS",P1=-2147483648,Qn="$$$$",Ln="M  V30 ",K8="M  V30 COUNTS ",Y8="M  V30 MDLV30/STEREL",X8="M  V30 MDLV30/STERAC",Lr="M  V30 END CTAB",_8=34028234663852886e22,Ur={l:2,m:0,h:4},_0=4194304,yr={l:4179966,m:4063288,h:7},W8={l:4194302,m:4193343,h:127},K1='" ',sr='stroke="',fP='stroke-width="',DP='" />',$8='class="event" ',a8='opacity="0" />',W0=16711680,ti="' position:",Ai="'. Position:",ei=1099511627776,ni=2199023255552,Pi=4398046511104,hr="', position:",ri="Assignment of aromatic double bonds failed",Un=2147483646,ii={102:1,4:1},Tr="Members of ESR groups must only be stereo centers with known configuration.",br="Over- or under-specified stereo feature or more than one racemic type bond",cr="Ambiguous configuration at stereo center because of 2 parallel bonds",fi={114:1,88:1,26:1,4:1},uP=1e5,yn=1.399999976158142,Se=3.1415927410125732,wr=1.5707963705062866,Di=.029999999329447746,ui=6.2831854820251465,oi=1.9106119871139526,li=2.2639999389648438,gr=2.16759991645813,Qi=2.0662999153137207,Li=2.3561999797821045,Ui=2.2844998836517334,yi=1.9106120066311025,si=2.094395160675049,sn=.6000000238418579,oP=1.7999999523162842,jr=.05000000074505806,P0={4:1,9:1,5:1,8:1,37:1},lP=.2617993950843811,Ve=2.0943951023931953,hn=1.2000000476837158,Br=1.4299999475479126,QP=1.659999966621399,dr=1.4600000381469727,mr=1.899999976158142,hi=1.8200000524520874,Ti=2.4600000381469727,bi=2.4200000762939453,LP=2.450000047683716,kr=2.440000057220459,ci=2.4000000953674316,Fr=2.319999933242798,vr=2.2899999618530273,wi=1.8600000143051147,gi=2.5299999713897705,ji=2.490000009536743,Bi=2.4700000286102295,di=2.880000114440918,mi=2.9200000762939453,ki=2.869999885559082,pr=2.8299999237060547,Fi=2.7899999618530273,vi=2.809999942779541,pi=2.799999952316284,Ri=3.049999952316284,Rr=.3199999928474426,Ci=1.3300000429153442,Tn=.8500000238418579,Mi=1.600000023841858,Hi=1.3899999856948853,Cr=1.159999966621399,qi=1.1100000143051147,Ei=1.0700000524520874,Si=1.9600000381469727,Vi=1.7100000381469727,Mr=1.3600000143051147,xi=1.340000033378601,zi=1.2200000286102295,Ni=1.2400000095367432,UP=1.2100000381469727,Ii=2.0999999046325684,Hr=1.850000023841858,qr=1.6299999952316284,Oi=1.4700000286102295,Gi=1.2799999713897705,Ji=1.440000057220459,Zi=1.7599999904632568,Ki=1.7300000190734863,Yi=1.7200000286102295,yP=1.6799999475479126,Xi=1.690000057220459,_i=1.6699999570846558,Er=1.649999976158142,bn=1.7000000476837158,Wi=1.6200000047683716,$i=1.3700000047683716,ai=1.2899999618530273,t9=1.4500000476837158,A9=1.5700000524520874,Sr=-0.5235987755982988,cn=-1.0471975511965976,e9=-0.7853981633974483,n9=.017453292519943295,P9=.17453292519943295,wn=1e-4,Ue={149:1},ye=57.29577951308232,r9=.021922173386725217,gn=1e-8,DA=2.5120960256267386,jn=1e-5,sP="MMFF94s+",i9="nonbonded cutoff",f9="dielectric constant",D9="dielectric model",u9="angle bend",o9="bond stretch",l9="electrostatic",Q9="out of plane",L9="stretch bend",U9="torsion angle",y9="van der waals",s9=.07000000000000006,h9=.1200000000000001,T9=-7.840000000000001,hP="OCL_RXN_V1.0:",b9="$RXN V3000",c9="'$MOL' tag not found",w9="'M  END' not found",g9="M  V30 BEGIN REACTANT",j9=`
Actelion Java MolfileCreator 2.0

  0  0  0  0  0  0              0 V3000
`,B9="M  V30 BEGIN PRODUCT",d9=2475109711874,m9=2475109711875,k9=2887426572290,F9=2887426572291,v9=2887627898882,p9=2887627898883,R9=2887627997186,C9=3437182386178,M9=3437182386179,H9=3437383712770,q9=3437383811074,E9=3437652148226,S9=3437652377602,V9=4674132967426,x9=4674334294018,z9=4674334392322,N9=4674602729474,I9=4674602958850,O9=5086449827842,G9=5086449827848,J9=5086651154434,Z9=5086651252738,K9=5086651252743,Y9=5086919589890,X9=5086919688194,_9=5636406968322,W9=5636407066626,$9=5636675502082,a9={l:589826,m:590112,h:4},tf={l:589827,m:590112,h:4},Af={l:589826,m:590160,h:4},ef={l:589827,m:590160,h:4},nf={l:688130,m:590160,h:4},Pf={l:589826,m:590224,h:4},rf={l:589827,m:590224,h:4},ff={l:819203,m:590224,h:4},Df={l:589826,m:590368,h:4},uf={l:688130,m:590368,h:4},of={l:819202,m:590368,h:4},lf={l:819208,m:590368,h:4},Qf={l:589826,m:590416,h:4},Lf={l:589832,m:590416,h:4},Uf={l:688130,m:590416,h:4},yf={l:819202,m:590416,h:4},sf={l:589826,m:598304,h:4},hf={l:589827,m:598304,h:4},Tf={l:688130,m:598304,h:4},bf={l:1114114,m:598304,h:4},cf={l:1212418,m:598304,h:4},wf={l:589826,m:598308,h:4},gf={l:589826,m:688416,h:4},jf={l:589826,m:688464,h:4},Bf={l:589826,m:688720,h:4},df={l:688130,m:688720,h:4},mf={l:589826,m:590112,h:288},kf={l:589826,m:590112,h:336},Ff={l:589826,m:688416,h:336},vf={l:589826,m:688464,h:336},pf={l:688130,m:688464,h:336},Rf={l:589826,m:590112,h:400},Cf={l:589826,m:688416,h:400},Mf={l:589826,m:688464,h:400},Hf={l:688130,m:688464,h:400},qf={l:589826,m:819488,h:400},Ef={l:589826,m:819536,h:400},Sf={l:589826,m:819600,h:400},Vf={l:688130,m:819600,h:400},xf={l:819202,m:819600,h:400},zf={l:589831,m:688416,h:592},Nf={l:589831,m:688464,h:592},If={l:688135,m:688464,h:592},Of={l:589831,m:819536,h:592},Gf={l:688135,m:819536,h:592},Jf={l:688135,m:819600,h:592},Zf={l:589832,m:1212704,h:592},Kf={l:589832,m:1212752,h:592},Yf={l:688136,m:1212752,h:592},Xf={l:589831,m:688464,h:656},_f={l:589826,m:590112,h:8480},Wf={l:589826,m:688416,h:8480},$f={l:589826,m:688464,h:8480},af={l:688130,m:688464,h:8480},tD={l:819202,m:819600,h:8480},AD={l:688135,m:1212752,h:8480},eD={l:589832,m:1213008,h:8480},nD={l:688136,m:1213008,h:8480},PD={l:589826,m:590112,h:8484},rD={l:589826,m:590160,h:8484},iD={l:688130,m:598304,h:8484},fD=.6262000203132629,DD=-1.3825000524520874,TP=.33169999718666077,uD=-1.4915000200271606,oD=.3540000021457672,lD=.38179999589920044,QD=-0.6019999980926514,LD=.07999999821186066,UD=-0.7379999756813049,yD="Atom-types are 64-bit numbers describing atoms and their near surrounding.",Vr="Recognized atom types and their contributions are:",sD="Druglikeness predictor not properly initialized.",hD=3.009999990463257,TD=-0.17000000178813934,bD=-0.1809999942779541,cD=-0.2029999941587448,Bn=2147483648,wD="ene-Metathesis",gD="Halogenation ring closure",jD=" (rootPairSets:",xr=-1e10,bP={204:1,4:1,9:1,5:1,8:1,30:1},cP=-99999,wA="any",BD="is not in a ring",zr="Excluded atoms:",wP="exactly 1",gP="exactly 2",Nr="less than 3",dD="less than 4",jP="at least 1",Ir="at least 2",Or="at least 3",mD="at least 4",Gr="1 or 2",kD="less than 2",FD="Allowed atoms:",se={138:1},vD="Atom Properties",BP=-8346432,pD="/html/editor/editor.html",RD="Structure Editor Help",dP="Dummy look and feel",xe="Substance Graphite",ze="Radiance Graphite",Ne="Radiance Night",CD=.6499999761581421,MD=.3499999940395355,Jr="editorButtons.png",HD="esrButtons.png",he=.800000011920929,Zr=-1774190527,Kr=-1438646166,Ie=-1433403286,mP=-1431655766,kP=-1448432982,vA=1073741866,Te=1073741930,Yr="MMFF94",Xr="MMFF94s",qD="/resources/forcefield/mmff94/94s/outofplane.csv",Oe="undefined",_r={172:1,4:1,5:1,8:1},ED=.08726646502812703,SD="overlapping",Wr=.003921568859368563,R0=1048575,$0=17592186044416,be=1e9,Ge=-17592186044416,g0=5.56,f0=11.12,pA=13.34,ce=14.44,dn=.3010299956639812,VD="BigInteger divide by zero",f1=4294967295,xD={l:0,m:0,h:524288},FP={32:1,115:1},$r={107:1,110:1},Je={36:1,43:1,38:1,85:1},vP=16777619,zD={36:1,38:1,85:1},pP="delete",RP=15525485,ND=5960464477539063e-23,ar=16777216,CP={4:1,32:1,56:1,80:1},t8="Invalid UTF8 sequence",A8=.20000000298023224,ID=" distance:",OD={4:1,5:1,11:1,8:1,452:1},GD="gGP`@dfyjidNcGI[WQCP`<",Ze={48:1},l={},T={},s1={},Gt={},w={},uA={},e={},y={},x={},Xt={},X={},gA={},K={},Pt={},r1={},e1={},nt={},At={},a={},E={},Tt={},d1={},m={},W={},C={},V1={},Lt={},v1={},j0={},Ct={},R={},S1={},l0={},Q0={},G0={},dt={},_t={},_={},$={},U,mn,MP;l.Heb=function(){};function Mu(f,t){typeof window===at&&typeof window.$gwt===at&&(window.$gwt[f]=t);}function Hu(f,t,A,n){l.ufb();var P=MP;function r(){for(var i=0;i<P.length;i++)P[i]();}if(f)try{Au(r)();}catch(i){f(t,i);}else Au(r)();}l.ufb=function(){MP==null&&(MP=[]);};function qu(){l.ufb();for(var f=MP,t=0;t<arguments.length;t++)f.push(arguments[t]);}l.sfb=function(){};function Ke(){}function Ye(f){return Array.isArray(f)&&f.xd===Ke?T.Uc(f):f.toString();}function z1(f,t){var A=d;if(f==="")return A;var n=f.split(".");if(!(n[0]in A)&&A.execScript&&A.execScript("var "+n[0]),t){var P=t.prototype.vd;P.e=t;}for(var r;n.length&&(r=n.shift());)A=A[r]=A[r]||!n.length&&t||{};return A;}l.Ffb=function(t){function A(){}return A.prototype=t||{},new A();},l.Efb=function(t){return t instanceof Array?t[0]:null;};function h0(f,t,A){var n=function(){return f.apply(n,arguments);};return t.apply(n,A),n;}function D1(){}function a0(f,t){for(var A in t)t[A].configurable=true;Object.defineProperties(f,t);}function N(f,t,A){var n=mn,P=n[f],r=l.Efb(P);P&&!r?U=P:(U=l.zfb(t),U.wd=A,!t&&(U.xd=Ke),n[f]=U);for(var i=3;i<arguments.length;++i)arguments[i].prototype=U;r&&(U.vd=r);}l.zfb=function(t){var A=t&&t.prototype;return !A&&(A=mn[t]),l.Ffb(A);},l.yfb=function(){},d.goog=d.goog||{},d.goog.global=d.goog.global||d,mn={},T.Jc=function(t,A){return F1(t)===F1(A);},T.Kc=function(t){return t.vd;},T.Lc=function(t){return w.Izb(t);};function JD(){}T.Oc=function(t,A){return RA(t)?T.kjb(t,A):we(t)?T.aib(t,A):kn(t)?T.Ugb(t,A):n8(t)?t.mb(A):e8(t)?T.Jc(t,A):s1.E4(t,A);},T.Qc=function(t){return RA(t)?T.ojb():we(t)?T.bib():kn(t)?T.Vgb():n8(t)?t.vd:e8(t)?T.Kc(t):s1.F4(t);},T.Sc=function(t){return RA(t)?T.pjb(t):we(t)?T.cib(t):kn(t)?T.Wgb(t):n8(t)?t.ob():e8(t)?T.Lc(t):s1.G4(t);},T.Uc=function(t){var A;return T.whb(T.Qc(t))+"@"+(A=w.aAb(w._zb(T.Sc(t))),A.toString(16));},N(1,null,{},JD),U.mb=function(t){return T.Jc(this,t);},U.nb=function(){return T.Kc(this);},U.ob=function(){return T.Lc(this);},U.pb=function(){var t;return T.whb(T.Qc(this))+"@"+(t=w.aAb(w._zb(T.Sc(this))),t.toString(16));},U.equals=function(f){return this.mb(f);},U.hashCode=function(){return this.ob();},U.toString=function(){return this.pb();},s1.E4=function(t,A){return s1.K4(t)?s1.H4(t,A):F1(t)===F1(A);},s1.F4=function(t){return t.vd||Array.isArray(t)&&F(l._ab,1)||l._ab;},s1.G4=function(t){return s1.L4(t)?s1.I4(t):w.Izb(t);},s1.H4=function(t,A){return t.equals(A);},s1.I4=function(t){return t.hashCode();},s1.J4=function(){return [];},s1.K4=function(t){return !!t&&!!t.equals;},s1.L4=function(t){return !!t&&!!t.hashCode;},s1.M4=function(t){return t.toString?t.toString():"[JavaScriptObject]";},Gt.q5=function(){Gt.q5=D1;var t,A;A=!Gt.z5(),t=new Gt.I5(),Gt.p5=A?new Gt.A5():t;},Gt.r5=function(t){Gt.q5(),Gt.p5.Mc(t);},Gt.s5=function(t){Gt.q5();var A;return A=Gt.p5.Nc(t),Gt.t5(A);},Gt.t5=function(t){var A,n,P,r;for(A="r5",n="ZB",r=d.Math.min(t.length,5),P=r-1;P>=0;P--)if(T.kjb(t[P].d,A)||T.kjb(t[P].d,n)){t.length>=P+1&&w.aAb(t).splice(0,P+1);break;}return t;},Gt.u5=function(t){var A=/function(?:\s+([\w$]+))?\s*\(/,n=A.exec(t);return n&&n[1]||g1;},Gt.v5=function(t){return Gt.q5(),t&&t[T1]?t[T1]:[];},Gt.w5=function(t){return Gt.q5(),t.name||(t.name=Gt.u5(t.toString()));},Gt.x5=function(t){return Gt.q5(),parseInt(t)||-1;},Gt.y5=function(t){Gt.q5();var A=t.backingJsObject;if(A&&A.stack){var n=A.stack,P=A+k1;return n.substring(0,P.length)==P&&(n=n.substring(P.length)),n.split(k1);}return [];},Gt.z5=function(){return Error.stackTraceLimit>0?(d.Error.stackTraceLimit=Error.stackTraceLimit=64,true):"stack"in new Error();},N(436,1,{}),Gt.A5=function(){},N(305,436,{},Gt.A5),U.Mc=function(t){var A={},n=[];t[T1]=n;for(var P=arguments.callee.caller;P;){var r=Gt.w5(P);n.push(r);var i=":"+r,D=A[i];if(D){var u,o;for(u=0,o=D.length;u<o;u++)if(D[u]===P)return;}(D||(A[i]=[])).push(P),P=P.caller;}},U.Nc=function(t){var A,n,P,r;for(P=Gt.v5(t),n=s1.S4(P),r=j(l.Ubb,i1,90,n,0,1),A=0;A<n;A++)r[A]=new T.$ib(s1.R4(P,A),null,-1);return r;},Gt.D5=function(t,A){var n,P,r,i,D,u,o,Q,L;return Q="",w.aAb(A).length==0?t.Oc(o0,g1,-1,-1):(L=T.Cjb(A),T.kjb(w.aAb(L).substr(0,3),"at ")&&(L=(w.Xzb(3,w.aAb(L).length+1),w.aAb(L).substr(3))),L=Gt.E5(L),D=w.aAb(L).indexOf("("),D==-1?(D=w.aAb(L).indexOf("@"),D==-1?(Q=L,L=""):(Q=T.Cjb((w.Xzb(D+1,w.aAb(L).length+1),w.aAb(L).substr(D+1))),L=T.Cjb((w.Wzb(0,D,w.aAb(L).length),w.aAb(L).substr(0,D))))):(n=w.aAb(L).indexOf(")",D),Q=(w.Wzb(D+1,n,w.aAb(L).length),w.aAb(L).substr(D+1,n-(D+1))),L=T.Cjb((w.Wzb(0,D,w.aAb(L).length),w.aAb(L).substr(0,D)))),D=T.rjb(L,Y1(46)),D!=-1&&(L=(w.Xzb(D+1,w.aAb(L).length+1),w.aAb(L).substr(D+1))),(w.aAb(L).length==0||T.kjb(L,"Anonymous function"))&&(L=g1),u=T.ujb(Q,Y1(58)),r=T.vjb(Q,Y1(58),u-1),o=-1,P=-1,i=o0,u!=-1&&r!=-1&&(i=(w.Wzb(0,r,w.aAb(Q).length),w.aAb(Q).substr(0,r)),o=Gt.x5((w.Wzb(r+1,u,w.aAb(Q).length),w.aAb(Q).substr(r+1,u-(r+1)))),P=Gt.x5((w.Xzb(u+1,w.aAb(Q).length+1),w.aAb(Q).substr(u+1)))),t.Oc(i,L,o,P));},Gt.E5=function(t){return t.replace(/\[.*?\]/g,"");},N(437,436,{}),U.Mc=function(t){},U.Oc=function(t,A,n,P){return new T.$ib(A,t+"@"+P,n<0?-1:n);},U.Nc=function(t){var A,n,P,r,i,D;if(r=Gt.y5(t),i=j(l.Ubb,i1,90,0,0,1),A=0,P=s1.S4(r),P==0)return i;for(D=Gt.D5(this,s1.R4(r,0)),T.kjb(D.d,g1)||(i[A++]=D),n=1;n<P;n++)i[A++]=Gt.D5(this,s1.R4(r,n));return i;},Gt.I5=function(){},N(306,437,{},Gt.I5),U.Oc=function(t,A,n,P){return new T.$ib(A,t,-1);},l.K5=function(t){return t;};function F(f,t){return l.M5(f,t);}l.M5=function(t,A){return T.Hhb(t,A);},l.N5=function(t){return t.__elementTypeCategory$==null?10:t.__elementTypeCategory$;},l.O5=function(t){return t.__elementTypeId$;};function y1(f,t,A,n,P,r){return l.Q5(f,t,A,n,P,0,r);}l.Q5=function(t,A,n,P,r,i,D){var u,o,Q,L,s;if(L=r[i],Q=i==D-1,u=Q?P:0,s=l.S5(u,L),P!=10&&v(F(t,D-i),A[i],n[i],u,s),!Q)for(++i,o=0;o<L;++o)l.U5(s,o,l.Q5(t,A,n,P,r,i,D));return s;};function j(f,t,A,n,P,r){var i;return i=l.S5(P,n),P!=10&&v(F(f,r),t,A,P,i),i;}l.S5=function(t,A){var n=new Array(A),P;switch(t){case 14:case 15:P=0;break;case 16:P=false;break;default:return n;}for(var r=0;r<A;++r)n[r]=P;return n;};function e8(f){return Array.isArray(f)&&f.xd===Ke;}l.U5=function(t,A,n){return t[A]=n;};function $1(f,t,A){return l.U5(f,t,A);}l.W5=function(t,A){t.vd=A;},l.X5=function(t,A){t.__elementTypeCategory$=A;},l.Y5=function(t,A){t.__elementTypeId$=A;};function v(f,t,A,n,P){return l.W5(P,f),P.wd=t,P.xd=Ke,l.Y5(P,A),l.X5(P,n),P;}l.$5=function(t,A){return l.N5(A)!=10&&v(T.Qc(A),A.wd,l.O5(A),l.N5(A),t),l.K5(t);};function Eu(f,t){return RA(f)?!!_D[t]:f.wd?!!f.wd[t]:we(f)?!!XD[t]:kn(f)?!!YD[t]:false;}function n8(f){return !Array.isArray(f)&&f.xd===Ke;}function Kt(f,t){return f!=null&&Eu(f,t);}function kn(f){return typeof f===u0;}function we(f){return typeof f===W1;}function ZD(f){return f!=null&&l.b7(f)&&f.xd!==Ke;}function KD(f,t){return l.c7(f,t);}function RA(f){return typeof f===q0;}l.b7=function(t){return typeof t===at||typeof t=="function";},l.c7=function(t,A){return t&&A&&t instanceof A;};function F1(f){return f??null;}function Rt(f){return Math.max(Math.min(f,St),-2147483648)|0;}var YD,XD,_D;l.Ieb=function(t){return t&&t.__java$exception;};function M1(f){var t;return Kt(f,27)?f:(t=l.Ieb(f),t||(t=new s1.y4(f),Gt.r5(t)),t);}function Ut(f){return f.backingJsObject;}function Su(f){var t;return f==null?false:(t=typeof f,T.kjb(t,u0)||T.kjb(t,W1)||T.kjb(t,q0)||f.$implements__java_io_Serializable||Array.isArray(f));}function Vu(f){var t;return t=typeof f,T.kjb(t,u0)||T.kjb(t,W1)||T.kjb(t,q0)?true:f!=null&&f.$implements__java_lang_Comparable;}T.Qgb=function(){T.Qgb=D1,Fn=w.aAb(false),w.aAb(true);},T.Rgb=function(t){return w.bAb((w.Qzb(t),t));},T.Sgb=function(t,A){return zu(w.bAb((w.Qzb(t),t)),w.bAb((w.Qzb(A),A)));},T.Tgb=function(t,A){return T.Sgb(t,A);},T.Ugb=function(t,A){return w.Qzb(t),F1(t)===F1(A);},T.Vgb=function(){return l.Cbb;},T.Wgb=function(t){return w.bAb((w.Qzb(t),t))?1231:1237;};function xu(f){return T.Qgb(),T.kjb(u0,typeof f);}function zu(f,t){return T.Qgb(),f==t?0:f?1:-1;}T.Zgb=function(t,A){return T.Qgb(),RA(t)?T.djb(t,A):we(t)?T.$hb(t,A):kn(t)?T.Tgb(t,A):t.Sb(A);},YD={4:1,300:1,32:1};var Fn;function Nu(f){return T.kjb(typeof f,q0)?true:f!=null&&f.$implements__java_lang_CharSequence;}T.uhb=function(t){t.i=Gu++;},T.vhb=function(t){t.n==null&&T.Lhb(t);},T.whb=function(t){return T.vhb(t),t.n;},T.xhb=function(t){return (t.f&4)!=0;},T.yhb=function(t){return (t.f&1)!=0;},T.zhb=function(){T.uhb(this),this.n=null,this.j=null,this.g=null,this.d=null,this.b=null,this.k=null,this.a=null;},T.Bhb=function(t){var A;return A=new T.zhb(),A.n="Class$"+(t?"S"+t:""+A.i),A.b=A.n,A.j=A.n,A;};function O(f){var t;return t=T.Bhb(f),T.Phb(f,t),t;}function GA(f,t){var A;return A=T.Bhb(f),T.Phb(f,A),A.f=t?8:0,A;}function Iu(){var f;return f=T.Bhb(null),f.f=2,f;}function Ae(f){var t;return t=T.Bhb(f),t.k=f,t.f=1,t;}T.Hhb=function(t,A){var n=t.a=t.a||[];return n[A]||(n[A]=t.Uc(A));};function Ou(f){if(f.Zc())return null;var t=f.k;return mn[t];}T.Lhb=function(t){if(t.Yc()){var A=t.c;A.Zc()?t.n="["+A.k:A.Yc()?t.n="["+A.Wc():t.n="[L"+A.Wc()+";",t.b=A.Vc()+"[]",t.j=A.Xc()+"[]";return;}var n=t.g,P=t.d;P=P.split("/"),t.n=T.Ohb(".",[n,T.Ohb("$",P)]),t.b=T.Ohb(".",[n,T.Ohb(".",P)]),t.j=P[P.length-1];},T.Ohb=function(t,A){for(var n=0;!A[n]||A[n]=="";)n++;for(var P=A[n++];n<A.length;n++)!A[n]||A[n]==""||(P+=t+A[n]);return P;},T.Phb=function(t,A){if(t){A.k=t;var P=Ou(A);if(!P){mn[t]=[A];return;}P.vd=A;}},N(207,1,{},T.zhb),U.Uc=function(t){var A;return A=new T.zhb(),A.f=4,t>1?A.c=T.Hhb(this,t-1):A.c=this,A;},U.Vc=function(){return T.vhb(this),this.b;},U.Wc=function(){return T.whb(this);},U.Xc=function(){return T.vhb(this),this.j;},U.Yc=function(){return T.xhb(this);},U.Zc=function(){return T.yhb(this);},U.pb=function(){return ((this.f&2)!=0?"interface ":(this.f&1)!=0?"":"class ")+(T.vhb(this),this.n);},U.f=0,U.i=0;var Gu=1;T.Uhb=function(t){return T.kjb(W1,typeof t)||KD(t,d.java.lang.Number$impl);},T.Vhb=function(t){if(T.Thb==null&&(T.Thb=new RegExp("^\\s*[+-]?(NaN|Infinity|((\\d+\\.?\\d*)|(\\.\\d+))([eE][+-]?\\d+)?[dDfF]?)\\s*$")),!T.Thb.test(t))throw Ut(new T.Zib(qA+t+'"'));return parseFloat(t);},T.Whb=function(t,A,n){var P,r,i,D,u;if(t==null)throw Ut(new T.Zib(rA));for(i=w.aAb(t).length,D=i>0&&(w.Xzb(0,w.aAb(t).length),w.aAb(t).charCodeAt(0)==45||(w.Xzb(0,w.aAb(t).length),w.aAb(t).charCodeAt(0)==43))?1:0,P=D;P<i;P++)if($u((w.Xzb(P,w.aAb(t).length),w.aAb(t).charCodeAt(P)))==-1)throw Ut(new T.Zib(qA+t+'"'));if(u=parseInt(t,10),r=u<A,isNaN(u))throw Ut(new T.Zib(qA+t+'"'));if(r||u>n)throw Ut(new T.Zib(qA+t+'"'));return u;},T.Xhb=function(t){return we(t)?T._hb(t):t.$c();},T.Yhb=function(t){return we(t)?T.dib(t):t._c();},N(120,1,{4:1,120:1}),T.Zhb=function(t,A){return JA(w.cAb((w.Qzb(t),t)),w.cAb((w.Qzb(A),A)));},T.$hb=function(t,A){return T.Zhb(t,A);},T._hb=function(t){return w.cAb((w.Qzb(t),t));},T.aib=function(t,A){return w.Qzb(t),F1(t)===F1(A);},T.bib=function(){return l.Fbb;},T.cib=function(t){return Rt(w.cAb((w.Qzb(t),t)));},T.dib=function(t){return Rt(w.cAb((w.Qzb(t),t)));};function Ju(f){return T.kjb(W1,typeof f);}function JA(f,t){return f<t?-1:f>t?1:f==t?f==0?JA(1/f,1/t):0:isNaN(f)?isNaN(t)?0:1:-1;}function HP(f){return isNaN(f)?{l:0,m:0,h:524160}:w.Yzb(f);}XD={4:1,32:1,301:1,120:1},T.GB=function(t){t.g=j(l.Ubb,i1,90,0,0,1);},T.HB=function(t){Gt.r5(t);},T.IB=function(t){return Gt.s5(t);},T.JB=function(t){return t.j&&(t.backingJsObject!==Vn&&t.jc(),t.g=null),t;},T.KB=function(t){return t.e;},T.LB=function(t,A){if(A instanceof Object)try{if(A.__java$exception=t,kt.userAgent.toLowerCase().indexOf("msie")!=-1&&ft.documentMode<9)return;var n=t;Object.defineProperties(A,{cause:{get:function(){var P=n.hc();return P&&P.fc();}},suppressed:{get:function(){return n.gc();}}});}catch{}},T.MB=function(t,A,n){var P,r,i,D,u;for(T.NB(t),r=(t.i==null&&(t.i=j(l.Zbb,i1,27,0,0,1)),t.i),i=0,D=r.length;i<D;++i)P=r[i],T.MB(P,A,"	"+n);u=t.e,u&&T.MB(u,A,n);},T.NB=function(t){var A,n,P;for(A=(t.g==null&&(t.g=T.IB(t)),t.g),n=0,P=A.length;n<P;++n);},T.OB=function(t,A){t.backingJsObject=A,T.LB(t,A);},T.PB=function(t){return T.QB(t,t.ic());},T.QB=function(t,A){var n;return n=T.whb(t.vd),A==null?n:n+": "+A;},T.RB=function(){T.GB(this),T.JB(this),this.jc();},T.SB=function(t){T.GB(this),this.f=t,T.JB(this),this.jc();},T.UB=function(t){if(!("stack"in t))try{throw t;}catch{}return t;};function Zu(f){var t;return f!=null&&(t=f.__java$exception,t)?t:KD(f,TypeError)?new T.Wib(f):new T.q4(f);}N(27,1,{4:1,27:1}),U.ec=function(t){return new Error(t);},U.fc=function(){return this.backingJsObject;},U.gc=function(){var t,A,n;for(n=(this.i==null&&(this.i=j(l.Zbb,i1,27,0,0,1)),this.i),A=j(l.Sbb,C1,1,n.length,5,1),t=0;t<n.length;t++)A[t]=n[t].backingJsObject;return A;},U.hc=function(){return T.KB(this);},U.ic=function(){return this.f;},U.jc=function(){T.OB(this,T.UB(this.ec(T.QB(this,this.f)))),T.HB(this);},U.pb=function(){return T.PB(this);},U.backingJsObject=Vn,U.j=true,T.aC=function(){T.RB.call(this);},T.bC=function(t){T.GB(this),T.JB(this),this.backingJsObject=t,T.LB(this,t),this.f=t==null?rA:Ye(t);},T.cC=function(t){T.SB.call(this,t);},N(19,27,{4:1,19:1,27:1},T.cC),T.dC=function(){T.aC.call(this);},T.eC=function(t){T.bC.call(this,t);},T.fC=function(t){T.cC.call(this,t);},N(33,19,m0),T.Lgb=function(){T.dC.call(this);},T.Mgb=function(t){T.fC.call(this,t);},N(79,33,m0,T.Lgb,T.Mgb),T.p4=function(){T.dC.call(this);},T.q4=function(t){T.eC.call(this,t);},T.r4=function(t){T.fC.call(this,t);},N(129,33,m0,T.q4),T.Vib=function(){T.p4.call(this);},T.Wib=function(t){T.q4.call(this,t);},T.Xib=function(t){T.r4.call(this,t);},N(69,129,m0,T.Vib,T.Wib,T.Xib),U.ec=function(t){return new TypeError(t);},T.cjb=function(t,A){return w.Xzb(A,w.aAb(t).length),w.aAb(t).charCodeAt(A);},T.djb=function(t,A){return T.ejb(t,A);},T.ejb=function(t,A){var n,P;return n=T._hb(w.aAb(t)),P=T._hb(w.aAb(A)),n==P?0:n<P?-1:1;},T.fjb=function(t,A){return T.ejb(w.aAb(t).toLowerCase(),w.aAb(A).toLowerCase());},T.gjb=function(t,A){return w.Qzb(t),t+(""+(w.Qzb(A),A));},T.hjb=function(t){var A;return T.Hjb(w.Azb(t,0,(A=t.length,w.xzb(),A)));},T.ijb=function(t){return T.Hjb(w.Azb(t,0,t.length));},T.jjb=function(t){var A;return A=w.aAb(ve).length,T.kjb(w.aAb(t).substr(w.aAb(t).length-A,A),ve);},T.kjb=function(t,A){return w.Qzb(t),F1(t)===F1(A);},T.ljb=function(t,A){return w.Qzb(t),A==null?false:T.kjb(t,A)?true:w.aAb(t).length==w.aAb(A).length&&T.kjb(w.aAb(t).toLowerCase(),w.aAb(A).toLowerCase());},T.mjb=function(t,A,n,P,r){w.Wzb(A,n,w.aAb(t).length),w.Wzb(r,r+(n-A),P.length),T.njb(t,A,n,P,r);},T.njb=function(t,A,n,P,r){for(;A<n;)P[r++]=T.cjb(t,A++);},T.ojb=function(){return l.Ybb;},T.pjb=function(t){var A,n;for(A=0,n=0;n<w.aAb(t).length;n++)A=(A<<5)-A+(w.Xzb(n,w.aAb(t).length),w.aAb(t).charCodeAt(n))|0;return A;},T.qjb=function(t,A,n){return T.sjb(t,Y1(A),n);},T.rjb=function(t,A){return w.aAb(t).indexOf(A);},T.sjb=function(t,A,n){return w.aAb(t).indexOf(A,n);};function Ku(f){return T.kjb(q0,typeof f);}T.ujb=function(t,A){return w.aAb(t).lastIndexOf(A);},T.vjb=function(t,A,n){return w.aAb(t).lastIndexOf(A,n);},T.wjb=function(t,A){var n;return n=T.xjb(A,"([/\\\\\\.\\*\\+\\?\\|\\(\\)\\[\\]\\{\\}$^])","\\\\$1"),T.xjb(t,n,"");},T.xjb=function(t,A,n){return n=T.Gjb(n),w.aAb(t).replace(new RegExp(A,"g"),n);},T.yjb=function(t,A){var n,P,r,i,D,u,o,Q;for(n=new RegExp(A,"g"),o=j(l.Ybb,st,2,0,6,1),P=0,Q=t,i=null;;)if(u=n.exec(Q),u==null||Q==""){o[P]=Q;break;}else D=u.index,o[P]=(w.Wzb(0,D,w.aAb(Q).length),w.aAb(Q).substr(0,D)),Q=T.Bjb(Q,D+w.aAb(w.aAb(u)[0]).length,w.aAb(Q).length),n.lastIndex=0,i==Q&&(o[P]=(w.Wzb(0,1,w.aAb(Q).length),w.aAb(Q).substr(0,1)),Q=(w.Xzb(1,w.aAb(Q).length+1),w.aAb(Q).substr(1))),i=Q,++P;if(w.aAb(t).length>0){for(r=o.length;r>0&&o[r-1]=="";)--r;r<o.length&&(w.aAb(o).length=r);}return o;},T.zjb=function(t,A){return T.kjb(w.aAb(t).substr(0,w.aAb(A).length),A);},T.Ajb=function(t,A){return w.Xzb(A,w.aAb(t).length+1),w.aAb(t).substr(A);},T.Bjb=function(t,A,n){return w.Wzb(A,n,w.aAb(t).length),w.aAb(t).substr(A,n-A);},T.Cjb=function(t){var A,n,P;for(n=w.aAb(t).length,P=0;P<n&&(w.Xzb(P,w.aAb(t).length),w.aAb(t).charCodeAt(P)<=32);)++P;for(A=n;A>P&&(w.Xzb(A-1,w.aAb(t).length),w.aAb(t).charCodeAt(A-1)<=32);)--A;return P>0||A<n?(w.Wzb(P,A,w.aAb(t).length),w.aAb(t).substr(P,A-P)):t;},T.Djb=function(t){return String.fromCharCode.apply(null,t);};function Y1(f){var t,A;return f>=q1?(t=55296+(f-q1>>10&1023)&n1,A=56320+(f-q1&1023)&n1,String.fromCharCode(t)+(""+String.fromCharCode(A))):String.fromCharCode(f&n1);}T.Fjb=function(t){var A,n,P,r,i;for(i=new y.Ovb(),n=t,P=0,r=n.length;P<r;++P)A=n[P],i.a?T.Sjb(i.a,i.b):i.a=new T.Wjb(i.d),T.Rjb(i.a,A);return i.a?w.aAb(i.e).length==0?i.a.a:i.a.a+(""+i.e):i.c;},T.Gjb=function(t){var A;for(A=0;0<=(A=w.aAb(t).indexOf("\\",A));)w.Xzb(A+1,w.aAb(t).length),w.aAb(t).charCodeAt(A+1)==36?t=(w.Wzb(0,A,w.aAb(t).length),w.aAb(t).substr(0,A)+"$"+T.Ajb(t,++A)):t=(w.Wzb(0,A,w.aAb(t).length),w.aAb(t).substr(0,A)+(""+T.Ajb(t,++A)));return t;},T.Hjb=function(t){return T.Ijb(t,0,t.length);},T.Ijb=function(t,A,n){var P,r,i,D;for(i=A+n,w.Wzb(A,i,t.length),D="",r=A;r<i;)P=d.Math.min(r+K0,i),D+=T.Djb(w.aAb(t).slice(r,P)),r=P;return D;},_D={4:1,206:1,32:1,2:1},T.Xjb=function(t){T.Mgb.call(this,t);},N(231,79,m0,T.Xjb),w.mzb=function(t){var A;return A=w.aAb(t).slice(),w.wzb(A,t);},w.nzb=function(t,A,n,P,r,i){var D,u,o,Q,L;if(r!=0)for(F1(t)===F1(n)&&(t=w.aAb(t).slice(A,A+r),A=0),o=w.aAb(n),u=A,Q=A+r;u<Q;)D=d.Math.min(u+K0,Q),r=D-u,L=w.aAb(t).slice(u,D),w.aAb(L).splice(0,0,w.aAb(P),w.aAb(i?r:0)),Array.prototype.splice.apply(o,L),u=D,P+=r;},w.ozb=function(t,A){return w.wzb(new Array(A),t);},w.pzb=function(t,A,n){w.aAb(t).splice(A,0,n);},w.qzb=function(t,A,n){w.nzb(n,0,t,A,n.length,false);},w.rzb=function(t,A){w.aAb(t).push(A);},w.szb=function(t,A){w.aAb(t).push(A);},w.tzb=function(t,A,n){w.aAb(t).splice(A,n);},w.uzb=function(t,A){w.aAb(t).length=A;},w.vzb=function(t,A){w.aAb(t).sort(A);},N(500,1,{}),w.Fzb=function(){},w.Gzb=function(t){switch(typeof t){case q0:return T.pjb(w.aAb(t));case W1:return T.cib(w.aAb(t));case u0:return T.Wgb(w.aAb(t));default:return t==null?0:w.Izb(t);}},w.Hzb=function(){return ++w.Ezb;},w.Izb=function(t){return t.$H||(t.$H=w.Hzb());},N(434,1,{},w.Fzb),w.Ezb=0,w.Jzb=function(t){if(!t)throw Ut(new T.iib());},w.Kzb=function(t,A){if(!t)throw Ut(new T.jib(A));},w.Lzb=function(t,A,n){if(t>A)throw Ut(new T.jib(ae+t+N8+A));if(t<0||A>n)throw Ut(new T.Ogb(ae+t+XP+A+I8+n));},w.Mzb=function(t){if(t<0)throw Ut(new T.Uib("Negative array size: "+t));},w.Nzb=function(t,A){if(t!=A)throw Ut(new y.wtb());},w.Ozb=function(t){if(!t)throw Ut(new y.Qub());},w.Pzb=function(t,A){if(t<0||t>=A)throw Ut(new T.Mgb(_P+t+WP+A));},w.Qzb=function(t){if(t==null)throw Ut(new T.Vib());return t;},w.Rzb=function(t,A){if(t==null)throw Ut(new T.Xib(A));},w.Szb=function(t,A){if(t<0||t>A)throw Ut(new T.Mgb(_P+t+WP+A));},w.Tzb=function(t,A,n){if(t<0||A>n)throw Ut(new T.Mgb(ae+t+XP+A+", size: "+n));if(t>A)throw Ut(new T.jib(ae+t+N8+A));},w.Uzb=function(t){if(!t)throw Ut(new T.kib());},w.Vzb=function(t){if(!t)throw Ut(new T.lib("Unable to add element to queue"));},w.Wzb=function(t,A,n){if(t<0||A>n||A<t)throw Ut(new T.Xjb(ae+t+XP+A+I8+n));},w.Xzb=function(t,A){if(t<0||t>=A)throw Ut(new T.Xjb(_P+t+WP+A));},w.Yzb=function(t){var A,n;return A=new ArrayBuffer(8),w.aAb(new Float64Array(A))[0]=t,n=w.aAb(new Uint32Array(A)),w.dAb(n[0]|0,n[1]|0);},w.Zzb=function(t,A){return t[A];},w.$zb=function(t){return t===void 0;},w._zb=function(t){return t>>>0;},w.aAb=function(t){return t;},w.bAb=function(t){return t;},w.cAb=function(t){return t;},N(435,1,{}),l.Sbb=O(1),l._ab=O(0),l.gbb=O(436),l.dbb=O(305),l.fbb=O(437),l.ebb=O(306),l.Cbb=O(300),l.Ebb=O(207),l.Rbb=O(120),l.Fbb=O(301),l.Zbb=O(27),l.Hbb=O(19),l.Tbb=O(33),l.Kbb=O(79),l.Mbb=O(129),l.Pbb=O(69),l.Ybb=O(2),l.Xbb=O(231),l.Zdb=O(434),l.$db=O(435),uA.Vc=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt;for(this.c=t.length,this.d=t[0].length,G=d.Math.min(this.c,this.d),this.e=j(l.h7,Ft,6,d.Math.min(this.c+1,this.d),15,1),this.a=y1(l.h7,[i1,Ft],[15,6],15,[this.c,G],2),this.b=y1(l.h7,[i1,Ft],[15,6],15,[this.d,this.d],2),r=j(l.h7,Ft,6,this.d,15,1),xt=j(l.h7,Ft,6,this.c,15,1),tt=d.Math.min(this.c-1,this.d),ot=d.Math.max(0,d.Math.min(this.d-2,this.c)),S=0;S<d.Math.max(tt,ot);S++){if(S<tt){for(this.e[S]=0,s=S;s<this.c;s++)this.e[S]=uA.Wc(this.e[S],t[s][S]);if(this.e[S]!=0){for(t[S][S]<0&&(this.e[S]=-this.e[S]),L=S;L<this.c;L++)t[L][S]/=this.e[S];t[S][S]+=1;}this.e[S]=-this.e[S];}for(B=S+1;B<this.d;B++){if(S<tt&this.e[S]!=0){for(bt=0,s=S;s<this.c;s++)bt+=t[s][S]*t[s][B];for(bt=-bt/t[S][S],L=S;L<this.c;L++)t[L][B]+=bt*t[L][S];}r[B]=t[S][B];}if(true&S<tt)for(L=S;L<this.c;L++)this.a[L][S]=t[L][S];if(S<ot){for(r[S]=0,s=S+1;s<this.d;s++)r[S]=uA.Wc(r[S],r[s]);if(r[S]!=0){for(r[S+1]<0&&(r[S]=-r[S]),h=S+1;h<this.d;h++)r[h]/=r[S];r[S+1]+=1;}if(r[S]=-r[S],S+1<this.c&r[S]!=0){for(h=S+1;h<this.c;h++)xt[h]=0;for(k=S+1;k<this.d;k++)for(b=S+1;b<this.c;b++)xt[b]+=r[k]*t[b][k];for(M=S+1;M<this.d;M++)for(bt=-r[M]/r[S+1],b=S+1;b<this.c;b++)t[b][M]+=bt*xt[b];}for(L=S+1;L<this.d;L++)this.b[L][S]=r[L];}}for(it=d.Math.min(this.d,this.c+1),tt<this.d&&(this.e[tt]=t[tt][tt]),this.c<it&&(this.e[it-1]=0),ot+1<it&&(r[ot]=t[ot][it-1]),r[it-1]=0,p=tt;p<G;p++){for(L=0;L<this.c;L++)this.a[L][p]=0;this.a[p][p]=1;}for(V=tt-1;V>=0;V--)if(this.e[V]!=0){for(g=V+1;g<G;g++){for(bt=0,s=V;s<this.c;s++)bt+=this.a[s][V]*this.a[s][g];for(bt=-bt/this.a[V][V],h=V;h<this.c;h++)this.a[h][g]+=bt*this.a[h][V];}for(b=V;b<this.c;b++)this.a[b][V]=-this.a[b][V];for(this.a[V][V]=1+this.a[V][V],L=0;L<V-1;L++)this.a[L][V]=0;}else {for(L=0;L<this.c;L++)this.a[L][V]=0;this.a[V][V]=1;}for(I=this.d-1;I>=0;I--){if(I<ot&r[I]!=0)for(g=I+1;g<G;g++){for(bt=0,s=I+1;s<this.d;s++)bt+=this.b[s][I]*this.b[s][g];for(bt=-bt/this.b[I+1][I],h=I+1;h<this.d;h++)this.b[h][g]+=bt*this.b[h][I];}for(L=0;L<this.d;L++)this.b[L][I]=0;this.b[I][I]=1;}for(Qt=it-1,u=d.Math.pow(2,-52);it>0;){for(q=it-2;q>=-1&&q!=-1;q--)if(d.Math.abs(r[q])<=u*(d.Math.abs(this.e[q])+d.Math.abs(this.e[q+1]))){r[q]=0;break;}if(q==it-2)Z=4;else {for(z=it-1;z>=q&&z!=q;z--)if(bt=(z!=it?d.Math.abs(r[z]):0)+(z!=q+1?d.Math.abs(r[z-1]):0),d.Math.abs(this.e[z])<=u*bt){this.e[z]=0;break;}z==q?Z=3:z==it-1?Z=1:(Z=2,q=z);}switch(++q,Z){case 1:for(o=r[it-2],r[it-2]=0,g=it-2;g>=q;g--)for(bt=uA.Wc(this.e[g],o),P=this.e[g]/bt,ct=o/bt,this.e[g]=bt,g!=q&&(o=-ct*r[g-1],r[g-1]=P*r[g-1]),L=0;L<this.d;L++)bt=P*this.b[L][g]+ct*this.b[L][it-1],this.b[L][it-1]=-ct*this.b[L][g]+P*this.b[L][it-1],this.b[L][g]=bt;break;case 2:for(o=r[q-1],r[q-1]=0,g=q;g<it;g++)for(bt=uA.Wc(this.e[g],o),P=this.e[g]/bt,ct=o/bt,this.e[g]=bt,o=-ct*r[g],r[g]=P*r[g],L=0;L<this.c;L++)bt=P*this.a[L][g]+ct*this.a[L][q-1],this.a[L][q-1]=-ct*this.a[L][g]+P*this.a[L][q-1],this.a[L][g]=bt;break;case 3:{for(ut=d.Math.max(d.Math.max(d.Math.max(d.Math.max(d.Math.abs(this.e[it-1]),d.Math.abs(this.e[it-2])),d.Math.abs(r[it-2])),d.Math.abs(this.e[q])),d.Math.abs(r[q])),yt=this.e[it-1]/ut,wt=this.e[it-2]/ut,D=r[it-2]/ut,lt=this.e[q]/ut,i=r[q]/ut,A=((wt+yt)*(wt-yt)+D*D)/2,n=yt*D*(yt*D),Dt=0,A!=0|n!=0&&(Dt=d.Math.sqrt(A*A+n),A<0&&(Dt=-Dt),Dt=n/(A+Dt)),o=(lt+yt)*(lt-yt)+Dt,Q=lt*i,g=q;g<it-1;g++){for(bt=uA.Wc(o,Q),P=o/bt,ct=Q/bt,g!=q&&(r[g-1]=bt),o=P*this.e[g]+ct*r[g],r[g]=P*r[g]-ct*this.e[g],Q=ct*this.e[g+1],this.e[g+1]=P*this.e[g+1],s=0;s<this.d;s++)bt=P*this.b[s][g]+ct*this.b[s][g+1],this.b[s][g+1]=-ct*this.b[s][g]+P*this.b[s][g+1],this.b[s][g]=bt;if(bt=uA.Wc(o,Q),P=o/bt,ct=Q/bt,this.e[g]=bt,o=P*r[g]+ct*this.e[g+1],this.e[g+1]=-ct*r[g]+P*this.e[g+1],Q=ct*r[g+1],r[g+1]=P*r[g+1],g<this.c-1)for(L=0;L<this.c;L++)bt=P*this.a[L][g]+ct*this.a[L][g+1],this.a[L][g+1]=-ct*this.a[L][g]+P*this.a[L][g+1],this.a[L][g]=bt;}r[it-2]=o;}break;case 4:{if(this.e[q]<=0)for(this.e[q]=this.e[q]<0?-this.e[q]:0,L=0;L<=Qt;L++)this.b[L][q]=-this.b[L][q];for(;q<Qt&&!(this.e[q]>=this.e[q+1]);){if(bt=this.e[q],this.e[q]=this.e[q+1],this.e[q+1]=bt,q<this.d-1)for(L=0;L<this.d;L++)bt=this.b[L][q+1],this.b[L][q+1]=this.b[L][q],this.b[L][q]=bt;if(q<this.c-1)for(L=0;L<this.c;L++)bt=this.a[L][q+1],this.a[L][q+1]=this.a[L][q],this.a[L][q]=bt;++q;}--it;}}}},uA.Wc=function(t,A){var n;return d.Math.abs(t)>d.Math.abs(A)?(n=A/t,n=d.Math.abs(t)*d.Math.sqrt(1+n*n)):A!=0?(n=t/A,n=d.Math.abs(A)*d.Math.sqrt(1+n*n)):n=0,n;},N(200,1,ie,uA.Vc),U.c=0,U.d=0,l.l7=O(200),e._c=function(){e._c=D1,e.Xc=v(F(l.j7,1),H,6,15,[0,WA,14286847,13402367,12779264,16758197,9474192,3166456,16715021,9494608,11789301,11230450,9109248,12560038,1578e4,16744448,16777008,2093087,8442339,9388244,4062976,15132390,12567239,10921643,9083335,10255047,14706227,15765664,5296208,13140019,8224944,12750735,6721423,12419299,16752896,10889513,6076625,7351984,Y0,9764863,9756896,7586505,5551541,3907230,2396047,687500,27013,12632256,16767375,10909043,6717568,10380213,13924864,9699476,4366e3,5707663,51456,7394559,16777159,14286791,13107143,10747847,9437127,6422471,4587463,3211207,2097095,65436,58997,54354,48952,43812,5096191,5089023,2200790,2522539,2516630,1528967,13684960,16765219,12105936,10900557,5724513,10375093,11230208,7688005,4358806,4325478,32e3,7384058,47871,41471,36863,33023,27647,5528818,7888099,9064419,10565332,11739092,11739066,11734438,12389767,13041766,13369433,13697103,14221381,14680120,15073326,15400998,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13158600,1334015,56540,15075850,15132160,56540,15075850,15461355,8553170,1016335,1016335,1334015,15132160,3289770,14456450,16422400,16422400,11819700,3289770,1016335]);},e.ad=function(t){t.v=new Tt.JN();},e.bd=function(t){var A,n;(t.G&32)==0&&(n=e.ew(t.L),n!=null&&(t.w.a==0&&t.w.b==0&&(A=t.R.c*e.Ul(t.L),t.v=e.Zd(t),e.od(t,A),e.Ud(t,null,A,0)),t.Cb(Rt(t.A)),t.L.O!=1&&e.Vd(t,448),t.tb(n,t.w.a,t.w.b+fe*t.A)));},e.cd=function(t,A){return t==null?A:A==null?t:t+","+A;},e.dd=function(t,A){e.ui(A,t.R),e.xi(A,t.v),e.wi(A,t.w);},e.ed=function(t){var A;A=t.R.c*(t.r!=0?t.r:e.Ul(t.L)),t.Y=A*e.Zc,t.T=A*e.Yc,t.S=A*.38,t.W=A*.47,t.X=Rt(A*t.J*e.$c+.5),t.V=A*.12,t.Z=A*.4,t.A=A*.5+.5;},e.fd=function(t){var A,n;for(t=J(t,EA),n=0;n<(E.sK(),E.rK).length;n++)if(H1(t,E.rK[n]))return E.qK[n];return A=new T.Wjb("R"),pt(J(t,QA),0)&&(A.a+="0"),pt(J(t,SA),0)&&(A.a+="3"),pt(J(t,VA),0)&&(A.a+="4"),pt(J(t,xA),0)&&(A.a+="5"),pt(J(t,zA),0)&&(A.a+="6"),pt(J(t,NA),0)&&(A.a+="7"),pt(J(t,De),0)&&(A.a+="8"),A.a;},e.gd=function(t,A,n,P){var r,i;r=new e.je(),i=new e.je(),r.a=A.a,r.c=A.c,r.b=(A.a+A.b)/2,r.d=(A.c+A.d)/2,i.a=r.b,i.c=r.d,i.b=A.b,i.d=A.d,e.Rd(t,r)&&(e.Vd(t,t.p[n]),t.qb(r)),e.Rd(t,i)&&(e.Vd(t,t.p[P]),t.qb(i)),e.Vd(t,t.Q);},e.hd=function(t,A,n,P){var r,i,D,u,o;u=(A.b-A.a)/10,o=(A.d-A.c)/10,r=new e.je(),e.Dm(t.L,e.Xo(t.L,n,P))?(i=-3,D=-3):(i=t.p[n],D=t.p[P]),e.Vd(t,i),r.a=A.a,r.c=A.c,r.b=A.a+u*2,r.d=A.c+o*2,t.qb(r),r.a=A.a+u*4,r.c=A.c+o*4,r.b=A.a+u*5,r.d=A.c+o*5,t.qb(r),e.Vd(t,D),r.a=A.a+u*5,r.c=A.c+o*5,r.b=A.a+u*6,r.d=A.c+o*6,t.qb(r),r.a=A.a+u*8,r.c=A.c+o*8,r.b=A.b,r.d=A.d,t.qb(r),e.Vd(t,t.Q);},e.jd=function(t,A,n){t.ub(A-t.V/2,n-t.V/2,t.V);},e.kd=function(t,A,n,P){e.Dm(t.L,e.Xo(t.L,n,P))?(e.Vd(t,-3),t.qb(A),e.Vd(t,t.Q)):t.p[n]!=t.p[P]?e.gd(t,A,n,P):t.p[n]!=0?(e.Vd(t,t.p[n]),t.qb(A),e.Vd(t,t.Q)):t.qb(A);},e.ld=function(t,A,n,P,r,i,D){D.a=A.a+r,D.c=A.c+i,D.b=A.b+r,D.d=A.d+i,e.kd(t,D,n,P);},e.md=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g;for(s=A.b-A.a,c=A.d-A.c,o=d.Math.sqrt(s*s+c*c),Q=2*Jt(c1(d.Math.round(o/(4*t.Y)))),h=s/(Q-1),g=c/(Q-1),e.Dm(t.L,e.Xo(t.L,n,P))?(r=-3,i=-3):(r=t.p[n],i=t.p[P]),L=A.a-t.Y/2,b=A.c-t.Y/2,e.Vd(t,r),u=0;u<(Q/2|0);u++)t.ub(L,b,t.Y),L+=h,b+=g;for(e.Vd(t,i),D=0;D<(Q/2|0);D++)t.ub(L,b,t.Y),L+=h,b+=g;e.Vd(t,t.Q);},e.nd=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k;o=(A.c-A.d)/9,b=(A.b-A.a)/9,Q=A.b+o,c=A.d+b,L=A.b-o,g=A.d-b,s=(A.a+Q)/2,B=(A.c+c)/2,h=(A.a+L)/2,k=(A.c+g)/2,D=new Tt.DN(3),u=new Tt.DN(4),Tt.yN(D,A.a,A.c),Tt.yN(D,s,B),Tt.yN(D,h,k),Tt.yN(u,h,k),Tt.yN(u,s,B),Tt.yN(u,Q,c),Tt.yN(u,L,g),e.Dm(t.L,e.Xo(t.L,n,P))?(r=-3,i=-3):(r=t.p[n],i=t.p[P],t.L.O!=1&&(i=e.qd(t,n),r==e.Cl(t.L,n)&&(r=i))),e.Vd(t,r),t.sb(D),e.Vd(t,i),t.sb(u),e.Vd(t,t.Q);},e.od=function(t,A){var n,P;for(P=0;P<t.$.a.length;P++)t.v=Tt.IN(t.v,y.xj(t.$,P));e.pd(t,A),n=.1*A,t.v.c-=n,t.v.d-=n,t.v.b+=2*n,t.v.a+=2*n;},e.pd=function(t,A){var n,P,r,i,D,u,o;for(r=j(l.Feb,ht,6,t.L.q,16,1),P=0;P<t.L.r;P++)e.Bm(t.L,P)&&(r[e.Zl(t.L,0,P)]=true,r[e.Zl(t.L,1,P)]=true);for(D=new Tt.JN(),n=0;n<t.L.q;n++)i=e.Jm(t.L,n)?A*.47:r[n]?A*.38:0,i!=0&&(u=e.Ai(t.R,e.Ql(t.L,n)),o=e.Bi(t.R,e.Rl(t.L,n)),Tt.HN(D,u-i,o-i,i*2,i*2),t.v=Tt.IN(t.v,D));},e.qd=function(t,A){var n,P;if((t.G&4224)!=0||(P=e.rd(t,A),P==-1&&(n=e.Fo(t.L,A),n!=-1&&(A=n,P=e.rd(t,A))),P==-1))return t.p[A];switch(P&255){case 1:return 384;case 2:return 64;default:return 448;}},e.rd=function(t,A){var n,P,r;return r=-1,P=-1,(t.G&128)!=0||(e.zm(t.L,A)&&(r=e.Hl(t.L,A),P=e.Gl(t.L,A)),n=e.Ho(t.L,A),n!=-1&&(r=e.cm(t.L,n),P=e.bm(t.L,n)),r!=-1&&r!=0&&(r|=P<<8)),r;},e.sd=function(t){var A,n,P,r;for(t.Ab(2*t.S),r=new e.je(),P=0;P<t.L.r;P++)A=e.Zl(t.L,0,P),n=e.Zl(t.L,1,P),e.Bm(t.L,P)&&(r.a=e.Ai(t.R,e.Ql(t.L,A)),r.c=e.Bi(t.R,e.Rl(t.L,A)),r.b=e.Ai(t.R,e.Ql(t.L,n)),r.d=e.Bi(t.R,e.Rl(t.L,n)),e.Vd(t,-2),t.qb(r));},e.td=function(t){var A,n,P,r,i,D;if(t.L.K){for(D=t.W,e.Vd(t,-7),A=0;A<t.L.f;A++)e.Jm(t.L,A)&&t.ub(e.Ai(t.R,e.Ql(t.L,A))-D,e.Bi(t.R,e.Rl(t.L,A))-D,2*D);for(t.Ab(2*t.W),i=new e.je(),r=0;r<t.L.r;r++)n=e.Zl(t.L,0,r),P=e.Zl(t.L,1,r),pt(J(J(e.Ol(t.L,n),e.Ol(t.L,P)),r0),0)&&(i.a=e.Ai(t.R,e.Ql(t.L,n)),i.c=e.Bi(t.R,e.Rl(t.L,n)),i.b=e.Ai(t.R,e.Ql(t.L,P)),i.d=e.Bi(t.R,e.Rl(t.L,P)),t.qb(i));}},e.ud=function(t){var A,n,P,r;if(t.L.K){if(e.Vd(t,320),(t.G&8)!=0)for(A=0;A<t.L.f;A++)pt(J(e.Ol(t.L,A),-536870913),0)&&t.ub(e.Ai(t.R,e.Ql(t.L,A))-t.Z/2,e.Bi(t.R,e.Rl(t.L,A))-t.Z/2,t.Z);for(r=0;r<t.L.g;r++)e.hm(t.L,r)!=0&&(n=e.Zl(t.L,0,r),P=e.Zl(t.L,1,r),t.ub((e.Ai(t.R,e.Ql(t.L,n))+e.Ai(t.R,e.Ql(t.L,P))-t.Z)/2,(e.Bi(t.R,e.Rl(t.L,n))+e.Bi(t.R,e.Rl(t.L,P))-t.Z)/2,t.Z));}},e.vd=function(t){t.J=1,t.R=new e.Ci(),t.$=new y.Mj(),t.U=new y.Mj(),t.s=j(l.Feb,ht,6,t.L.q,16,1),t.w=new Tt.vN(),t.Q=0,t.C=-1,e._d(t);},e.wd=function(t,A){var n;if(e.Lo(t.L,A)==0)return  false;for(n=0;n<e.Lo(t.L,A);n++)if(!e.Dm(t.L,e._o(t.L,A,n)))return  false;return  true;},e.xd=function(t){var A;for(t.q=j(l.Feb,ht,6,t.L.q,16,1),A=0;A<t.L.r;A++)t.q[e.Zl(t.L,0,A)]=true,t.q[e.Zl(t.L,1,A)]=true;},e.yd=function(t,A){var n;if(e.$o(t.L,A)!=2)return  false;for(n=0;n<2;n++)if(e.ap(t.L,A,n)!=2)return  false;return  true;},e.zd=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c;for(h=false,r.a=0,r.b=0,P>0?i=$P:i=aP,c=e.Yl(t.L,A,n),L=0;L<e.$o(t.L,A);L++)D=e._o(t.L,A,L),u=c,e.Zl(t.L,0,D)==A?s=e.Zl(t.L,1,D):s=e.Zl(t.L,0,D),s!=n&&(b=e.Yl(t.L,A,s),u<b&&(u+=I1),o=u-b,P>0?(o<A1&&(h=true),o>$P&&(o=$P),o<.523598776&&(o=.523598776),o<=i&&(i=o,Q=t.T*d.Math.tan(i-k0)/2,r.a=-(Q*d.Math.sin(u)),r.b=-(Q*d.Math.cos(u)))):(o>=A1&&(h=true),o<aP&&(o=aP),o>5.759586531&&(o=5.759586531),o>=i&&(i=o,Q=t.T*d.Math.tan(4.712388981-i)/2,r.a=-(Q*d.Math.sin(u)),r.b=-(Q*d.Math.cos(u)))));return h;},e.Ad=function(t,A,n,P){var r;if(A==0){n<0?P.a=t.T:P.a=-t.T,P.b=0;return;}r=d.Math.atan(n/A),A<0&&(r+=A1),P.a=-(t.T*d.Math.sin(r)),P.b=t.T*d.Math.cos(r);},e.Bd=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g;r=new e.je(),o=new e.je(),L=new Tt.vN(),Q=new Tt.vN(),D=e.Zl(t.L,0,n),u=e.Zl(t.L,1,n),P&&(h=A.a,A.a=A.b,A.b=h,h=A.c,A.c=A.d,A.d=h,b=D,D=u,u=b),e.Rd(t,A)&&(e.gq(t.L,n)?(r.a=A.a,r.c=A.c,r.b=A.b,r.d=A.d,s=P?-e.Dp(t.L,n):e.Dp(t.L,n),s==0&&(s=1),e.Ad(t,A.b-A.a,A.d-A.c,L),s>0?(o.a=A.a+L.a,o.c=A.c+L.b,o.b=A.b+L.a,o.d=A.d+L.b,(e.zd(t,D,u,1,Q)||e.$o(t.L,D)>1)&&(o.a+=Q.a+L.b,o.c+=Q.b-L.a)):(o.a=A.a-L.a,o.c=A.c-L.b,o.b=A.b-L.a,o.d=A.d-L.b,(e.zd(t,D,u,-1,Q)||e.$o(t.L,D)>1)&&(o.a+=Q.a+L.b,o.c+=Q.b-L.a)),e.im(t.L,n)==386&&e.Qd(r,o),e.Rd(t,r)&&e.kd(t,r,D,u),e.im(t.L,n)==8?e.Rd(t,o)&&e.hd(t,o,D,u):e.Rd(t,o)&&e.kd(t,o,D,u)):(e.Ad(t,A.b-A.a,A.d-A.c,L),c=L.a/2,g=L.b/2,i=false,r.a=A.a+c,r.c=A.c+g,r.b=A.b+c,r.d=A.d+g,e.$o(t.L,D)>1&&(e.zd(t,D,u,1,Q)?(r.a+=Q.a,r.c+=Q.b,e.$o(t.L,D)==2&&(Q.a!=0||Q.b!=0)&&(r.a+=L.b,r.c-=L.a)):t.o[D]=new Tt.wN(r.a,r.c)),o.a=A.a-c,o.c=A.c-g,o.b=A.b-c,o.d=A.d-g,e.$o(t.L,D)>1&&(e.zd(t,D,u,0,Q)?(o.a+=Q.a,o.c+=Q.b,e.$o(t.L,D)==2&&(Q.a!=0||Q.b!=0)&&(o.a+=L.b,o.c-=L.a)):(t.o[D]=new Tt.wN(o.a,o.c),i=true)),e.im(t.L,n)==386&&e.Qd(r,o),e.im(t.L,n)==8?i?(e.hd(t,r,D,u),e.kd(t,o,D,u)):(e.kd(t,r,D,u),e.hd(t,o,D,u)):(e.kd(t,r,D,u),e.kd(t,o,D,u))));},e.Cd=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c;for(L=t.Q,s=t.F,(t.G&I0)!=0&&(t.Q=-6,t.F=xn,e.Vd(t,1)),t.o=j(l.fab,C1,55,t.L.q,0,1),o=0;o<t.L.r;o++)(e.im(t.L,o)==2||e.im(t.L,o)==386||e.im(t.L,o)==8)&&e.Fd(t,o);for(Q=0;Q<t.L.r;Q++)e.im(t.L,Q)!=2&&e.im(t.L,Q)!=386&&e.im(t.L,Q)!=8&&e.Fd(t,Q);if((t.G&64)==0)for(u=0;u<t.L.r;u++)e.am(t.L,u)!=0&&(r=null,e.am(t.L,u)==1||e.am(t.L,u)==2?(e.fm(t.L,u)==2||e.cm(t.L,u)==0||A[e.cm(t.L,u)][e.bm(t.L,u)]>1)&&(e.am(t.L,u)==1?r=e.fm(t.L,u)==2?"E":e.Fm(t.L,u)?"p":"P":r=e.fm(t.L,u)==2?"Z":e.Fm(t.L,u)?"m":"M"):r="?",r!=null&&(t.Cb((t.X*2+1)/3|0),e.Vd(t,e.Dm(t.L,u)?-3:t.L.O==1||(t.G&Z1)!=0?t.Q:448),n=e.Zl(t.L,0,u),P=e.Zl(t.L,1,u),b=(e.Ai(t.R,e.Ql(t.L,n))+e.Ai(t.R,e.Ql(t.L,P)))/2,c=(e.Bi(t.R,e.Rl(t.L,n))+e.Bi(t.R,e.Rl(t.L,P)))/2,i=(e.Ai(t.R,e.Ql(t.L,n))-e.Ai(t.R,e.Ql(t.L,P)))/3,D=(e.Bi(t.R,e.Rl(t.L,n))-e.Bi(t.R,e.Rl(t.L,P)))/3,e.Id(t,b+D,c-i,r,true),e.Vd(t,t.Q),t.Cb(t.X)));if((t.G&4)!=0){for(t.Cb((t.X*2+1)/3|0),e.Vd(t,384),u=0;u<t.L.r;u++)n=e.Zl(t.L,0,u),P=e.Zl(t.L,1,u),h=e.Yp(t.L,u)?"d":e.Up(t.L,u)?"a":"",b=(e.Ai(t.R,e.Ql(t.L,n))+e.Ai(t.R,e.Ql(t.L,P)))/2,c=(e.Bi(t.R,e.Rl(t.L,n))+e.Bi(t.R,e.Rl(t.L,P)))/2,e.Id(t,b,c,h+(""+u),true);e.Vd(t,t.Q),t.Cb(t.X);}(t.G&I0)!=0&&(t.Q=L,t.F=s);},e.Dd=function(t){var A,n;for(n=new y.Bqb(t.U);n.a<n.c.a.length;)A=y.Aqb(n),e.Vd(t,A.a),e.jd(t,A.b,A.c);e.Vd(t,t.Q);},e.Ed=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt;if(t.K||t.yb(A,e.Il(t.L,A),e.Ai(t.R,e.Ql(t.L,A)),e.Bi(t.R,e.Rl(t.L,A))),yt=null,e.Bl(t.L,A)!=0&&(Zt=d.Math.abs(e.Bl(t.L,A))==1?"":""+d.Math.abs(e.Bl(t.L,A)),yt=e.Bl(t.L,A)<0?Zt+"-":Zt+"+"),G=null,wt=e.Ol(t.L,A),et(wt,0)!=0&&(pt(J(wt,tn),0)&&(G=G==null?"*":G+",*"),pt(J(wt,An),0)&&(G=G==null?"!*":G+",!*"),pt(J(wt,en),0)?G=G==null?"ha":G+",ha":pt(J(wt,2),0)?G=G==null?"a":G+",a":pt(J(wt,4),0)&&(G=G==null?"!a":G+",!a"),pt(J(wt,Z1),0)&&(G=G==null?"s":G+",s"),pt(J(wt,X0),0)&&(V=J(wt,X0),et(V,1792)==0?G=G==null?"h0":G+",h0":et(V,1664)==0?G=G==null?"h1":G+",h1":et(V,1408)==0?G=G==null?"h2":G+",h2":et(V,128)==0?G=G==null?"h>0":G+",h>0":et(V,384)==0?G=G==null?"h>1":G+",h>1":et(V,896)==0?G=G==null?"h>2":G+",h>2":et(V,i0)==0?G=G==null?"h<3":G+",h<3":et(V,1536)==0?G=G==null?"h<2":G+",h<2":et(V,1152)==0&&(G=G==null?"h1-2":G+",h1-2")),pt(J(wt,mA),0)&&(o=J(wt,mA),et(o,nn)==0?G=G==null?"c0":G+",c0":et(o,ue)==0?G=G==null?"c+":G+",c+":et(o,oe)==0&&(G=G==null?"c-":G+",c-")),pt(J(wt,$A),0)&&(ct=J(wt,$A),et(ct,98304)==0?G=G==null?"pi0":G+",pi0":et(ct,81920)==0?G=G==null?"pi1":G+",pi1":et(ct,49152)==0?G=G==null?"pi2":G+",pi2":et(ct,I0)==0&&(G=G==null?"pi>0":G+",pi>0")),pt(J(wt,aA),0)&&(lt=J(wt,aA),et(lt,Pn)==0?G=G==null?"n1":G+",n1":et(lt,pe)==0?G=G==null?"n2":G+",n2":et(lt,Re)==0?G=G==null?"n3":G+",n3":et(lt,rn)==0?G=G==null?"n<3":G+",n<3":et(lt,v0)==0?G=G==null?"n<4":G+",n<4":et(lt,te)==0?G=G==null?"n>1":G+",n>1":et(lt,iA)==0?G=G==null?"n>2":G+",n>2":et(lt,le)==0?G=G==null?"n>3":G+",n>3":et(lt,3276800)==0?G=G==null?"n1-2":G+",n1-2":et(lt,2228224)==0?G=G==null?"n1-3":G+",n1-3":et(lt,2490368)==0&&(G=G==null?"n2-3":G+",n2-3")),pt(J(wt,Qe),0)&&(c=J(wt,Qe),et(c,zn)==0?G=G==null?"e0":G+",e0":et(c,Nn)==0?G=G==null?"e1":G+",e1":et(c,In)==0?G=G==null?"e2":G+",e2":et(c,On)==0?G=G==null?"e3":G+",e3":et(c,tr)==0?G=G==null?"e<2":G+",e<2":et(c,Ar)==0?G=G==null?"e<3":G+",e<3":et(c,fn)==0?G=G==null?"e<4":G+",e<4":et(c,Ce)==0?G=G==null?"e>0":G+",e>0":et(c,Gn)==0?G=G==null?"e>1":G+",e>1":et(c,Jn)==0?G=G==null?"e>2":G+",e>2":et(c,Dn)==0?G=G==null?"e>3":G+",e>3":et(c,er)==0?G=G==null?"e1-2":G+",e1-2":et(c,nr)==0?G=G==null?"e1-3":G+",e1-3":et(c,Pr)==0&&(G=G==null?"e2-3":G+",e2-3")),pt(J(wt,120),0)&&(bt=J(wt,120),et(bt,112)==0?G=G==null?"!r":G+",!r":et(bt,8)==0?G=G==null?"r":G+",r":et(bt,96)==0?G=G==null?"rb<3":G+",rb<3":et(bt,104)==0?G=G==null?"rb2":G+",rb2":et(bt,88)==0?G=G==null?"rb3":G+",rb3":et(bt,56)==0&&(G=G==null?"rb4":G+",rb4")),pt(J(wt,kA),0)&&(G=G==null?"r"+y0(t0(J(wt,kA),22)):G+","+("r"+y0(t0(J(wt,kA),22)))),pt(J(wt,EA),0)&&(G=e.cd(G,e.fd(wt))),pt(J(wt,IA),0)&&(G=G==null?"f":G+",f")),e.Ml(t.L,A)!=0&&(G=e.cd(G,""+e.Ml(t.L,A))),qt=0,e.Pl(t.L,A)!=0)switch(e.Pl(t.L,A)){case 16:yt=yt==null?"|":yt+",|";break;case 32:qt=1;break;case 48:qt=2;}if(s=null,(t.G&64)==0){if(e.wm(t.L,A))s="?";else if(e.Al(t.L,A)!=0&&(e.Hl(t.L,A)==0||n==null||n[e.Hl(t.L,A)][e.Gl(t.L,A)]>1))if(e.$o(t.L,A)==2)switch(e.Al(t.L,A)){case 2:s=e.ym(t.L,A)?"p":"P";break;case 1:s=e.ym(t.L,A)?"m":"M";break;default:s="*";}else switch(e.Al(t.L,A)){case 1:s=e.ym(t.L,A)?"r":"R";break;case 2:s=e.ym(t.L,A)?"s":"S";break;default:s="*";}}if((t.G&768)!=0&&(s=e.cd(s,""+e.mw(t.L,A))),ut=null,(t.G&16)!=0&&e.Ll(t.L,A)!=0&&(ut=""+e.Ll(t.L,A)),B=null,e.Ip(t.L,A)!=-1&&(g=e.rd(t,A),g!=-1&&(B=g==0?"abs":((g&255)==1?"&":"or")+(1+(g>>8)))),I=0,(t.G&L0)==0&&(t.L.K?pt(J(e.Ol(t.L,A),j1),0)&&(I=e.op(t.L,A)):(e.Tl(t.L,A)!=6||e.Ml(t.L,A)!=0||!t.q[A]||e.Pl(t.L,A)!=0)&&(I=e.op(t.L,A))),ot=false,tt=false,i=(t.G&q1)!=0?null:e.El(t.L,A),i!=null&&T.kjb(w.aAb(i).substr(0,1),"]")&&(ot=G==null,tt=true,G=e.cd((w.Xzb(1,w.aAb(i).length+1),w.aAb(i).substr(1)),G),i=null),i!=null?I=0:e.Jl(t.L,A)!=null?(r=pt(J(e.Ol(t.L,A),1),0)?"[!":"[",i=r+e.Kl(t.L,A)+"]",w.aAb(i).length>5&&(i=r+e.Jl(t.L,A).length+"]"),pt(J(e.Ol(t.L,A),j1),0)&&(I=-1)):pt(J(e.Ol(t.L,A),1),0)?(i="?",pt(J(e.Ol(t.L,A),j1),0)&&(I=-1)):(e.Tl(t.L,A)!=6||yt!=null||G!=null&&(!ot||(t.G&LA)==0)||I>0||!t.q[A])&&(i=e.Il(t.L,A)),Qt=0,!e.Pm(t.L,A)&e.Jm(t.L,A)&&e.Vd(t,-8),i!=null?(Qt=t.vb(i),e.Id(t,e.Ai(t.R,e.Ql(t.L,A)),e.Bi(t.R,e.Rl(t.L,A)),i,true),t.s[A]=true):e.yd(t,A)&&e.Hd(t,e.Ai(t.R,e.Ql(t.L,A)),e.Bi(t.R,e.Rl(t.L,A)),A),yt!=null&&(t.Cb((t.X*2+1)/3|0),Nt=e.Ai(t.R,e.Ql(t.L,A))+((Qt+t.vb(yt))/2+1),Bt=e.Bi(t.R,e.Rl(t.L,A))-((t.wb()*4-4)/8|0),e.Id(t,Nt,Bt,yt,true),t.Cb(t.X)),(t.G&2)!=0&&(G=""+A),G!=null&&(tt?t.Cb((t.X*5+1)/6|0):t.Cb((t.X*2+1)/3|0),Nt=e.Ai(t.R,e.Ql(t.L,A))-(Qt+t.vb(G))/2,Bt=e.Bi(t.R,e.Rl(t.L,A))-((t.wb()*4-4)/8|0),e.Id(t,Nt,Bt,G,true),t.Cb(t.X)),s!=null&&(t.Cb((t.X*2+1)/3|0),Nt=e.Ai(t.R,e.Ql(t.L,A))-(Qt+t.vb(s))/2,Bt=e.Bi(t.R,e.Rl(t.L,A))+((t.wb()*4+4)/8|0),Et=t.C,t.L.O!=1&&(t.G&Z1)==0&&e.Vd(t,448),e.Id(t,Nt,Bt,s,false),e.Vd(t,Et),t.Cb(t.X)),ut!=null&&(t.Cb((t.X*2+1)/3|0),Nt=e.Ai(t.R,e.Ql(t.L,A))+((Qt+t.vb(ut))/2+1),Bt=e.Bi(t.R,e.Rl(t.L,A))+((t.wb()*4+4)/8|0),Et=t.C,e.Vd(t,e.Am(t.L,A)?384:448),e.Id(t,Nt,Bt,ut,true),e.Vd(t,Et),t.Cb(t.X)),B!=null&&(P=e.Md(t,A),t.Cb((t.X*2+1)/3|0),Nt=e.Ai(t.R,e.Ql(t.L,A))+.7*t.wb()*d.Math.sin(P),Bt=e.Bi(t.R,e.Rl(t.L,A))+.7*t.wb()*d.Math.cos(P),Et=t.C,!t.K&&t.L.O!=1&&e.Vd(t,e.qd(t,A)),e.Id(t,Nt,Bt,B,false),e.Vd(t,Et),t.Cb(t.X)),I==0&&qt==0){t.C==-8&&e.Vd(t,-9);return;}for(q=j(l.h7,Ft,6,4,15,1),z=0;z<e.Mo(t.L,A);z++)for(u=e._o(t.L,A,z),it=0;it<2;it++)e.Zl(t.L,it,u)==A&&(xt=e.Yl(t.L,e.Zl(t.L,it,u),e.Zl(t.L,1-it,u)),xt<un?(q[0]-=xt+k0,q[3]+=xt+A1):xt<0?(q[2]+=xt+k0,q[3]-=xt):xt<k0?(q[1]+=xt,q[2]+=k0-xt):(q[0]+=xt-k0,q[1]+=A1-xt));if(e.$o(t.L,A)==0?e.Hm(t.L,A)?q[3]-=.2:q[1]-=.2:q[1]-=.1,(yt!=null||ut!=null)&&(q[1]+=10),(G!=null||s!=null)&&(q[3]+=10),M="",I!=0&&(S=t.vb("H"),p=0,k=t.wb(),I==-1?(M="n",t.Cb((t.X*2+1)/3|0),p=t.vb(M)):I>1&&(M=""+I,t.Cb((t.X*2+1)/3|0),p=t.vb(M)),q[1]<.6||q[3]<.6?(L=e.Bi(t.R,e.Rl(t.L,A)),q[1]<=q[3]?(q[1]+=10,Q=e.Ai(t.R,e.Ql(t.L,A))+(Qt+S)/2):(q[3]+=10,Q=e.Ai(t.R,e.Ql(t.L,A))-(Qt+S)/2-p)):(Q=e.Ai(t.R,e.Ql(t.L,A)),q[0]<q[2]?(q[0]+=10,L=e.Bi(t.R,e.Rl(t.L,A))-k):(q[2]+=10,L=e.Bi(t.R,e.Rl(t.L,A))+k)),p>0&&(Nt=Q+(S+p)/2,Bt=L+((t.wb()*4+4)/8|0),e.Id(t,Nt,Bt,M,true),t.Cb(t.X)),e.Id(t,Q,L,"H",true)),D=0,qt!=0){for(Dt=50,h=0,Z=0;Z<4;Z++)b=Z>1?Z-2:Z+2,q[Z]<Dt?(D=Z,Dt=q[Z],h=q[b]):q[Z]==Dt&&q[b]>h&&(D=Z,h=q[b]);switch(D){case 0:Q=e.Ai(t.R,e.Ql(t.L,A)),L=e.Bi(t.R,e.Rl(t.L,A))-t.V-Qt/2;break;case 1:Q=e.Ai(t.R,e.Ql(t.L,A))+t.V+Qt/2,L=e.Bi(t.R,e.Rl(t.L,A));break;case 2:Q=e.Ai(t.R,e.Ql(t.L,A)),L=e.Bi(t.R,e.Rl(t.L,A))+t.V+Qt/2;break;default:Q=e.Ai(t.R,e.Ql(t.L,A))-t.V-Qt/2,L=e.Bi(t.R,e.Rl(t.L,A));}if(qt==1)y.sj(t.$,new Tt.KN(Q-t.V,L-t.V,2*t.V,2*t.V)),t.K||y.sj(t.U,new e.ie(Q,L,e.wd(t,A)?-3:t.p[A]));else {switch(D){case 2:case 0:It=2*t.V,jt=0,Q-=t.V;break;case 1:It=0,jt=2*t.V,L-=t.V;break;default:It=0,jt=2*t.V,L-=t.V;}y.sj(t.$,new Tt.KN(Q-t.V,L-t.V,2*t.V,2*t.V)),t.K||y.sj(t.U,new e.ie(Q,L,e.wd(t,A)?-3:t.p[A])),y.sj(t.$,new Tt.KN(Q+It-t.V,L+jt-t.V,2*t.V,2*t.V)),t.K||y.sj(t.U,new e.ie(Q+It,L+jt,e.wd(t,A)?-3:t.p[A]));}}t.C==-8&&e.Vd(t,-9);},e.Fd=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;if(p=new e.je(),n=new e.je(),D=new e.je(),B=new Tt.vN(),g=new Tt.vN(),r=e.Zl(t.L,0,A),i=e.Zl(t.L,1,A),t.zb(A,e.Ai(t.R,e.Ql(t.L,r)),e.Bi(t.R,e.Rl(t.L,r)),e.Ai(t.R,e.Ql(t.L,i)),e.Bi(t.R,e.Rl(t.L,i))),!e.Pm(t.L,r)&&!e.Pm(t.L,i)&&pt(J(rt(e.Ol(t.L,r),e.Ol(t.L,i)),r0),0)&&e.Vd(t,-8),t.o[r]?(p.a=t.o[r].a,p.c=t.o[r].b):(p.a=e.Ai(t.R,e.Ql(t.L,r)),p.c=e.Bi(t.R,e.Rl(t.L,r))),t.o[i]?(p.b=t.o[i].a,p.d=t.o[i].b):(p.b=e.Ai(t.R,e.Ql(t.L,i)),p.d=e.Bi(t.R,e.Rl(t.L,i))),(e.hm(t.L,A)&UA)!=0){e.Rd(t,p)&&t.rb(p),e.Vd(t,-9);return;}switch(u=e.im(t.L,A)==8?0:e.im(t.L,A)==16?1:e.fm(t.L,A),u){case 1:if(o=e.im(t.L,A),(t.G&128)!=0&&(o==257||o==129)&&(M=e.Zl(t.L,0,A),b=e.Hl(t.L,M),b!=0)){for(h=e.Gl(t.L,M),s=0,P=0;P<t.L.f;P++)e.Hl(t.L,P)==b&&e.Gl(t.L,P)==h&&++s;s==1&&(o=1);}switch(o){case 1:e.Rd(t,p)&&e.kd(t,p,r,i);break;case 257:e.Od(t,p,r,i);break;case 129:for(q=p.b-p.a,S=p.d-p.c,e.Dm(t.L,e.Xo(t.L,r,i))?(Q=-3,L=-3):(Q=t.p[r],L=e.qd(t,r),Q==e.Cl(t.L,r)&&(Q=L)),c=2;c<17;c+=2)n.a=p.a+c*q/17-c*S/128,n.c=p.c+c*S/17+c*q/128,n.b=p.a+c*q/17+c*S/128,n.d=p.c+c*S/17-c*q/128,e.Rd(t,n)&&(e.Vd(t,c<9?Q:L),t.qb(n),e.Vd(t,t.Q));break;case 16:e.Rd(t,p)&&e.md(t,p,r,i);}break;case 0:case 2:if((t.s[r]||e.Qo(t.L,r)==2)&&(t.s[i]||e.Qo(t.L,i)==2)&&!e.gq(t.L,A)&&u==2){if(!e.Rd(t,p))break;e.Ad(t,p.b-p.a,p.d-p.c,B),q=B.a/2,S=B.b/2,n.a=p.a+q,n.c=p.c+S,n.b=p.b+q,n.d=p.d+S,D.a=p.a-q,D.c=p.c-S,D.b=p.b-q,D.d=p.d-S,e.im(t.L,A)==386&&e.Qd(n,D),e.kd(t,n,r,i),u==2?e.kd(t,D,r,i):e.hd(t,D,r,i);}else (t.s[i]||e.Qo(t.L,i)==2)&&u==2?e.Bd(t,p,A,false):(t.s[r]||e.Qo(t.L,r)==2)&&u==2?e.Bd(t,p,A,true):(k=e.Dp(t.L,A),k==0&&(k=1),n.a=p.a,n.c=p.c,n.b=p.b,n.d=p.d,e.Ad(t,p.b-p.a,p.d-p.c,B),k>0?(D.a=p.a+B.a,D.c=p.c+B.b,D.b=p.b+B.a,D.d=p.d+B.b,(e.zd(t,r,i,1,g)||e.$o(t.L,r)>1)&&(D.a+=g.a+B.b,D.c+=g.b-B.a),(e.zd(t,i,r,-1,g)||e.$o(t.L,i)>1)&&(D.b+=g.a-B.b,D.d+=g.b+B.a)):(D.a=p.a-B.a,D.c=p.c-B.b,D.b=p.b-B.a,D.d=p.d-B.b,(e.zd(t,r,i,-1,g)||e.$o(t.L,r)>1)&&(D.a+=g.a+B.b,D.c+=g.b-B.a),(e.zd(t,i,r,1,g)||e.$o(t.L,i)>1)&&(D.b+=g.a-B.b,D.d+=g.b+B.a)),e.im(t.L,A)==386&&e.Qd(n,D),e.Rd(t,n)&&e.kd(t,n,r,i),u==2?e.Rd(t,D)&&e.kd(t,D,r,i):e.Rd(t,D)&&e.hd(t,D,r,i));break;case 3:e.Rd(t,p)&&(e.kd(t,p,r,i),e.Ad(t,p.b-p.a,p.d-p.c,B),e.ld(t,p,r,i,B.a,B.b,n),e.ld(t,p,r,i,-B.a,-B.b,n));break;case 4:e.Rd(t,p)&&(e.Ad(t,p.b-p.a,p.d-p.c,B),e.ld(t,p,r,i,1.5*B.a,1.5*B.b,n),e.ld(t,p,r,i,.5*B.a,.5*B.b,n),e.ld(t,p,r,i,-0.5*B.a,-0.5*B.b,n),e.ld(t,p,r,i,-1.5*B.a,-1.5*B.b,n));break;case 5:e.Rd(t,p)&&(e.kd(t,p,r,i),e.Ad(t,p.b-p.a,p.d-p.c,B),e.ld(t,p,r,i,2*B.a,2*B.b,n),e.ld(t,p,r,i,B.a,B.b,n),e.ld(t,p,r,i,-B.a,-B.b,n),e.ld(t,p,r,i,-2*B.a,-2*B.b,n));}t.C==-8&&e.Vd(t,-9);},e.Gd=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g;for(b=false,P=0;P<t.L.g;P++)Q=null,e.Cm(t.L,P)?(s=e._l(t.L,P),L=e.$l(t.L,P),Q=s==L?"["+s+"]":"["+s+":"+L+"]"):(e.hm(t.L,P)&on)!=0?Q=(e.hm(t.L,P)&on)==v0?"a":(e.hm(t.L,P)&384)==256?"r!a":"!a":(e.hm(t.L,P)&384)!=0&&(Q=(e.hm(t.L,P)&384)==256?"r":"!r"),h=(e.hm(t.L,P)&iA)>>17,h!=0&&(Q=(Q??"")+h),Q!=null&&(A=e.Zl(t.L,0,P),n=e.Zl(t.L,1,P),b||(t.Cb((t.X*2+1)/3|0),b=true),c=(e.Ai(t.R,e.Ql(t.L,A))+e.Ai(t.R,e.Ql(t.L,n)))/2,g=(e.Bi(t.R,e.Rl(t.L,A))+e.Bi(t.R,e.Rl(t.L,n)))/2,i=e.Ai(t.R,e.Ql(t.L,n))-e.Ai(t.R,e.Ql(t.L,A)),D=e.Bi(t.R,e.Rl(t.L,n))-e.Bi(t.R,e.Rl(t.L,A)),r=d.Math.sqrt(i*i+D*D),o=.6*t.vb(Q),u=.55*t.wb(),r!=0&&(i>0?e.Id(t,c+o*D/r,g-u*i/r,Q,true):e.Id(t,c-o*D/r,g+u*i/r,Q,true)));b&&t.Cb(t.X);},e.Hd=function(t,A,n,P){y.sj(t.$,new Tt.KN(A-t.V,n-t.V,2*t.V,2*t.V)),t.K||y.sj(t.U,new e.ie(A,n,e.wd(t,P)?-3:t.p[P]));},e.Id=function(t,A,n,P,r){var i,D,u;r&&(i=t.vb(P),D=i/2+(t.wb()/8|0),u=t.wb()/2|0,(P=="+"||P=="-")&&(u=u*2/3),y.sj(t.$,new Tt.KN(A-D,n-u,2*D,2*u))),t.K||t.tb(P,A,n);},e.Jd=function(t){var A;A=t.a,t.a=t.b,t.b=A,A=t.c,t.c=t.d,t.d=A;},e.Kd=function(t,A,n){var P;return P=A==0?I1+t[0]-t[t.length-1]:t[A]-t[A-1],n>rr&&n<b0?P-=2*d.Math.cos(n+Me):P-=.5*d.Math.cos(n+Me),P;},e.Ld=function(t){var A;return A=new Tt.JN(),t.a<=t.b?(A.c=t.a,A.b=t.b-t.a):(A.c=t.b,A.b=t.a-t.b),t.c<=t.d?(A.d=t.c,A.a=t.d-t.c):(A.d=t.d,A.a=t.c-t.d),A;},e.Md=function(t,A){var n,P,r,i,D,u,o;for(n=j(l.h7,Ft,6,e.Lo(t.L,A),15,1),r=0;r<e.Lo(t.L,A);r++)n[r]=e.Yl(t.L,A,e.Zo(t.L,A,r));for(w.vzb(n,w.aAb(h0(y.frb.prototype.kd,y.frb,[]))),i=e.Nd(n,0),D=e.Kd(n,0,i),P=1;P<n.length;P++)u=e.Nd(n,P),o=e.Kd(n,P,u),D<o&&(D=o,i=u);return i;},e.Nd=function(t,A){var n;return A>0?(t[A]+t[A-1])/2:(n=A1+(t[0]+t[t.length-1])/2,n>A1?n-I1:n);},e.Od=function(t,A,n,P){var r,i,D,u;if(u=new e.je(),!(A.a==A.b&&A.c==A.d)){for(u.a=A.a,u.c=A.c,u.b=A.b,u.d=A.d,D=e.Ld(u),r=0;r<t.$.a.length;r++)if(i=y.xj(t.$,r),!(i.c>D.c+D.b||i.d>D.d+D.a||D.c>i.c+i.b||D.d>i.d+i.a)){if(e.Pd(t,u.a,u.c,r)){if(e.Pd(t,u.b,u.d,r))return;e.Sd(t,u,0,r),e.Od(t,u,n,P);return;}if(e.Pd(t,u.b,u.d,r)){e.Sd(t,u,1,r),e.Od(t,u,n,P);return;}}e.nd(t,u,n,P);}},e.Pd=function(t,A,n,P){var r;return (t.G&1)!=0?false:(r=y.xj(t.$,P),A>r.c&&A<r.c+r.b&&n>r.d&&n<r.d+r.a);},e.Qd=function(t,A){var n;n=t.b,t.b=A.b,A.b=n,n=t.d,t.d=A.d,A.d=n;},e.Rd=function(t,A){var n,P,r,i,D,u;if(A.a==A.b&&A.c==A.d){for(D=new y.Bqb(t.$);D.a<D.c.a.length;)if(i=y.Aqb(D),Tt.FN(i,A.a,A.c))return  false;return  true;}for(u=e.Ld(A),n=false,A.a>A.b&&(e.Jd(A),n=true),P=0;P<t.$.a.length;P++)if(i=y.xj(t.$,P),!(i.c>u.c+u.b||i.d>u.d+u.a||u.c>i.c+i.b||u.d>i.d+i.a)){if(e.Pd(t,A.a,A.c,P))return e.Pd(t,A.b,A.d,P)?(n&&e.Jd(A),false):(e.Sd(t,A,0,P),r=e.Rd(t,A),n&&e.Jd(A),r);if(e.Pd(t,A.b,A.d,P))return e.Sd(t,A,1,P),r=e.Rd(t,A),n&&e.Jd(A),r;}return n&&e.Jd(A),true;},e.Sd=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c;n==0?(s=A.a,b=A.c,h=A.b,c=A.d):(s=A.b,b=A.d,h=A.a,c=A.c),L=y.xj(t.$,P),o=h>s?L.c+L.b:L.c,Q=c>b?L.d+L.a:L.d,r=h-s,i=c-b,d.Math.abs(r)>d.Math.abs(i)?b==c?(D=o,u=b):(D=s+r*(Q-b)/i,h>s==o>D?u=Q:(D=o,u=b+i*(o-s)/r)):s==h?(D=s,u=Q):(u=b+i*(o-s)/r,c>b==Q>u?D=o:(D=s+r*(Q-b)/i,u=Q)),n==0?(A.a=D,A.c=u):(A.b=D,A.d=u);},e.Td=function(t,A){var n,P,r,i,D,u;if(t.L.q!=0){for(e.bw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),t.B=A,e.ed(t),r=e.gw(t.L),i=false,t.p=j(l.j7,H,6,t.L.q,15,1),n=0;n<t.L.q;n++)t.p[n]=e.Cl(t.L,n),t.p[n]!=0&&(i=true),e.Pm(t.L,n)&&(t.p[n]=128),e.rm(t.L,n)&&(t.G&j1)==0&&(t.p[n]=256);for(e.Vd(t,-10),e.td(t),e.sd(t),e.ud(t),e.bd(t),t.Cb(t.X),t.Ab(t.Y),e.Vd(t,t.Q),e.xd(t),w.uzb(t.U.a,0),w.uzb(t.$.a,0),(t.G&1)!=0&&(e.Cd(t,r),e.Dd(t),e.Gd(t)),D=0;D<t.L.q;D++)e.wd(t,D)?(e.Vd(t,-3),e.Ed(t,D,r),e.Vd(t,t.Q)):t.p[D]!=0?(e.Vd(t,t.p[D]),e.Ed(t,D,r),e.Vd(t,t.Q)):!i&&t.L.O!=1&&e.Tl(t.L,D)!=1&&e.Tl(t.L,D)!=6&&(t.G&i0)==0&&e.Jl(t.L,D)==null&&e.Tl(t.L,D)<e.Xc.length&&e.Xc[e.Tl(t.L,D)]!=0?(u=e.Xc[e.Tl(t.L,D)],e.Yd(t,(P=t.M!=0?t.M:t.D!=0?t.D:-1,Lt.T3(u,P))),e.Ed(t,D,r),e.Vd(t,t.Q)):e.Ed(t,D,r);(t.G&1)==0&&(e.Dd(t),e.Gd(t),e.Cd(t,r));}},e.Ud=function(t,A,n,P){var r;switch(r=n/2,P&Q1){case Q1:if(A){t.w.a=A.c+A.b/2,t.w.b=A.d+A.a-r;break;}case 0:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d+t.v.a+r,A&&t.w.b>A.d+A.a-r&&(t.w.b=A.d+A.a-r);break;case yA:if(A){t.w.a=A.c+A.b/2,t.w.b=A.d+r;break;}case sA:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d-r,A&&t.w.b<A.d+r&&(t.w.b=A.d+r);}},e.Vd=function(t,A){if(!t.K&&(A==-10&&(t.C=-999,A=t.Q),A!=-2&&A!=-7&&t.N!=0&&(A=-4),A!=t.C&&!(t.C==-8&&A!=-9)))switch(A==-8&&(t.O=t.C),A==-9&&(A=t.O),t.C=A,A){case 0:t.Bb(t.F==0?O0:t.F);break;case  -6:t.Bb(t.F);break;case  -4:t.Bb(t.N);break;case  -2:t.Bb(t.t);break;case  -3:t.Bb(t.u);break;case  -7:t.Bb(t.H);break;case  -8:t.Bb(t.I);break;case  -5:t.Bb(t.P);break;case 64:t.Bb(-14655233);break;case 128:t.Bb(fA);break;case 256:t.Bb(-4194049);break;case 192:t.Bb(-16711936);break;case 320:t.Bb(-24576);break;case 384:t.Bb(-16744448);break;case 448:t.Bb(-6291456);break;case 1:t.Bb(xn);break;default:t.Bb(O0);}},e.Wd=function(t,A){t.G=A;},e.Xd=function(t,A,n){t.Q=-6,t.F=A,t.D=n,e._d(t);},e.Yd=function(t,A){if(t.N!=0){t.C!=-4&&(t.C=-4,t.Bb(t.N));return;}t.C=-5,t.P=A,t.Bb(A);},e.Zd=function(t){var A,n,P,r,i;for(r=e.Ai(t.R,e.Ql(t.L,0)),n=e.Ai(t.R,e.Ql(t.L,0)),i=e.Bi(t.R,e.Rl(t.L,0)),P=e.Bi(t.R,e.Rl(t.L,0)),A=0;A<t.L.q;A++)r>e.Ai(t.R,e.Ql(t.L,A))&&(r=e.Ai(t.R,e.Ql(t.L,A))),n<e.Ai(t.R,e.Ql(t.L,A))&&(n=e.Ai(t.R,e.Ql(t.L,A))),i>e.Bi(t.R,e.Rl(t.L,A))&&(i=e.Bi(t.R,e.Rl(t.L,A))),P<e.Bi(t.R,e.Rl(t.L,A))&&(P=e.Bi(t.R,e.Rl(t.L,A)));return new Tt.KN(r,i,n-r,P-i);},e.$d=function(t,A,n){var P,r;return t.L.q==0?null:(t.v=e.Zd(t),P=t.R.c*e.Ul(t.L),r=new e.Ei(t.v,A,P,n),r.c==1&&r.a==0&&r.b==0?r=null:(e.ui(r,t.R),e.xi(r,t.v)),e.Ud(t,A,P,n),r);},e._d=function(t){var A;A=t.M!=0?t.M:t.D!=0?t.D:-1,t.t=Lt.U3(A),t.u=Lt.T3(Zn,A),t.H=-24321,t.I=-6291392;},e.ae=function(t,A,n,P){var r;return e.be(t,A,n,P),e.zi(t.R)?null:(r=t.R,e.vi(t.R,t.L),t.R=new e.Ci(),r);},e.be=function(t,A,n,P){var r,i,D,u;if(t.L.q==0)return null;for(D=t.Db(n,P),e.bw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),e.xd(t),w.uzb(t.U.a,0),w.uzb(t.$.a,0),t.B=A,e.ed(t),t.Cb(t.X),t.K=true,i=0;i<t.L.q;i++)e.Ed(t,i,null);return t.K=false,r=t.R.c*e.Ul(t.L),e.od(t,r),e.Ud(t,n,r,P),!n||Tt.GN(n,t.v)?D:(u=new e.Ei(t.v,n,r,P),e.ui(u,t.R),e.xi(u,t.v),e.wi(u,t.w),D?(e.ui(u,D),D):u);},e.ce=function(t){e.de.call(this,t,0);},e.de=function(t,A){e.ad(this),this.L=t,this.G=A,this.xb();},N(121,1,{121:1}),U.xb=function(){e.vd(this);},U.yb=function(t,A,n,P){},U.zb=function(t,A,n,P,r){},U.Db=function(t,A){return e.$d(this,t,A);},U.r=0,U.t=0,U.u=0,U.A=0,U.C=0,U.D=0,U.F=0,U.G=0,U.H=0,U.I=0,U.J=0,U.K=false,U.M=0,U.N=0,U.O=0,U.P=0,U.Q=0,U.S=0,U.T=0,U.V=0,U.W=0,U.X=0,U.Y=0,U.Z=0,e.Yc=.15,e.Zc=.06,e.$c=.6,l.o7=O(121),e.ie=function(t,A,n){this.b=t,this.c=A,this.a=n;},N(144,1,{144:1},e.ie),U.a=0,U.b=0,U.c=0,l.m7=O(144),e.je=function(){},N(67,1,{},e.je),U.a=0,U.b=0,U.c=0,U.d=0,l.n7=O(67),e.ke=function(t,A){var n;if(t.i==null)return  false;for(n=0;n<t.i.length;n++)if(!A.Hb(t.i[n].a,t.i[n].b))return  false;return  true;},e.le=function(t,A,n){var P;if(t.i!=null)for(P=0;P<t.i.length;P++)t.i[P].a+=A,t.i[P].b+=n;},e.me=function(t,A){var n;if(t.i!=null)for(n=0;n<t.i.length;n++)t.i[n].a*=A,t.i[n].b*=A;},e.ne=function(t,A){t.g=A;},e.oe=function(t,A,n){var P;if(t.i!=null)for(P=0;P<t.i.length;P++)t.i[P].a=t.o[P]+A-t.k,t.i[P].b=t.p[P]+n-t.n;},e.pe=function(t,A,n){var P,r;if(t.k=A,t.n=n,t.i!=null)for(r=t.i.length,t.o=j(l.h7,Ft,6,r,15,1),t.p=j(l.h7,Ft,6,r,15,1),P=0;P<r;P++)t.o[P]=t.i[P].a,t.p[P]=t.i[P].b;},e.qe=function(t,A,n){var P,r,i;if(t.i!=null)for(P=0;P<t.i.length;P++)i=t.p[P]*A,r=t.o[P]-n,t.i[P].a=t.k+i*d.Math.sin(r),t.i[P].b=t.n+i*d.Math.cos(r);},e.re=function(t,A,n){var P,r,i,D;if(t.k=A,t.n=n,t.i!=null)for(D=t.i.length,t.o=j(l.h7,Ft,6,D,15,1),t.p=j(l.h7,Ft,6,D,15,1),i=0;i<D;i++)P=A-t.i[i].a,r=n-t.i[i].b,t.p[i]=d.Math.sqrt(P*P+r*r),t.o[i]=e.no(A,n,t.i[i].a,t.i[i].b);},N(193,1,{193:1}),U.Jb=function(t){var A,n;if(this.i==null)return null;for(A=new Tt.KN(this.i[0].a,this.i[0].b,0,0),n=1;n<this.i.length;n++)A.c>this.i[n].a?(A.b+=A.c-this.i[n].a,A.c=this.i[n].a):A.b<this.i[n].a-A.c&&(A.b=this.i[n].a-A.c),A.d>this.i[n].b?(A.a+=A.d-this.i[n].b,A.d=this.i[n].b):A.a<this.i[n].b-A.d&&(A.a=this.i[n].b-A.d);return A;},U.Nb=function(t){e.me(this,t);},U.pb=function(){var t;return t=new T.Mjb(),T.Ljb(t,ir+this.Lb()+'"'+this.Kb()+Kn),t.a;},U.Ob=function(t,A){e.oe(this,t,A);},U.Pb=function(t,A){e.qe(this,t,A);},U.Qb=function(t,A){e.re(this,t,A);},U.g=false,U.j=false,U.k=0,U.n=0,l.p7=O(193),e.ye=function(t,A,n){if(t.f[A])return 0;if(e.Tl(t.j,A)==7){if(e.Lo(t.j,A)==3)return 6;if(e.$o(t.j,A)==2)return t.a?0:4;}else {if(e.Tl(t.j,A)==8)return 10;if(e.Tl(t.j,A)==15||e.Tl(t.j,A)==33){if(e.$o(t.j,A)==3)return 8;}else if(e.Tl(t.j,A)==16||e.Tl(t.j,A)==34||e.Tl(t.j,A)==52){if(e.$o(t.j,A)==2)return 11;if(e.$o(t.j,A)==3)return e.Bl(t.j,A)==1?12:(n&&e.jn(t.j,A,1),5);}else if(e.Tl(t.j,A)==6)return e.Bl(t.j,A)==-1?e.Lo(t.j,A)==3?16:e.Mo(t.j,A)==3?15:14:(n&&e.jn(t.j,A,-1),e.Lo(t.j,A)!=e.Mo(t.j,A)?2:3);}return 0;},e.ze=function(t,A,n){if(t.f[A])return 0;if(t.a){if(e.Lo(t.j,A)!=3)return 0;}else if(e.Lo(t.j,A)>3)return 0;return e.Tl(t.j,A)==6?(n&&e.jn(t.j,A,1),2):e.Tl(t.j,A)==5?4:0;},e.Ae=function(t,A,n){if(e.Bl(t.j,A)!=0)return 0;if(t.a){if(e.Tl(t.j,A)==5)return e.vp(t.j,A)!=2?0:(n&&e.jn(t.j,A,1),1);if(e.Tl(t.j,A)==7)return e.vp(t.j,A)!=2?0:(n&&e.jn(t.j,A,-1),e.De(t,A)?6:3);if(e.Tl(t.j,A)==8)return e.vp(t.j,A)!=1?0:(n&&e.jn(t.j,A,-1),e.De(t,A)?7:4);if(e.Tl(t.j,A)==16)return e.vp(t.j,A)!=1?0:(n&&e.jn(t.j,A,-1),e.De(t,A)?5:2);if(e.Tl(t.j,A)==34)return e.vp(t.j,A)!=1?0:(n&&e.jn(t.j,A,-1),e.De(t,A)?4:1);}else {if(e.Tl(t.j,A)==5)return e.vp(t.j,A)>2?0:(n&&e.jn(t.j,A,1),1);if(e.Tl(t.j,A)==7)return e.vp(t.j,A)>2?0:(n&&e.jn(t.j,A,-1),e.De(t,A)?5:3);if(e.Tl(t.j,A)==8)return e.vp(t.j,A)>1?0:(n&&e.jn(t.j,A,-1),e.De(t,A)?7:4);if(e.Tl(t.j,A)==16)return e.vp(t.j,A)>1?0:(n&&e.jn(t.j,A,-1),e.De(t,A)?5:2);}return 0;},e.Be=function(t,A,n){var P,r,i;if(P=e.Tl(t.j,A),P>=5&&P<=8||P==15||P==16||P==33||P==34||P==52){if(i=e.pp(t.j,A),i!=0)return  true;if(r=e.Bl(t.j,A),P==5&&r>=0)return n&&e.jn(t.j,A,r-1),true;if(P!=5&&r<=0)return n&&e.jn(t.j,A,r+1),true;}return  false;},e.Ce=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c;for(A=0;A<t.j.f;A++)if(t.d[A])for(D=false,u=j(l.j7,H,6,t.j.f,15,1),c=j(l.j7,H,6,t.j.f,15,1),o=j(l.j7,H,6,t.j.f,15,1),u[0]=A,c[A]=-1,o[A]=1,r=0,Q=0;r<=Q&&!D;){for(i=u[r],L=0;L<e.$o(t.j,i)&&!D;L++)if(s=(o[i]&1)==1^e.fm(t.j,e._o(t.j,i,L))>1,P=e.Zo(t.j,i,L),o[P]==0&&s)if(t.d[P]){if((o[i]&1)==1){for(t.d[A]=false,t.d[P]=false,t.k+=2,b=i,h=0;h<o[i];h++)n=e.Xo(t.j,P,b),t.e[n]&&(t.e[n]=false,--t.c),e.In(t.j,n,e.fm(t.j,n)==1?2:e.fm(t.j,n)-1),P=b,b=c[P];D=true;}}else u[++Q]=P,c[P]=i,o[P]=o[i]+1;++r;}},e.De=function(t,A){var n;for(n=0;n<e.$o(t.j,A);n++)if(e.Lm(t.j,e.Zo(t.j,A,n)))return  true;return  false;},e.Ee=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;if(A!=null)t.e=A;else for(t.e=j(l.Feb,ht,6,t.j.g,16,1),h=0;h<t.j.g;h++)e.im(t.j,h)==8&&(t.e[h]=true,e.Mn(t.j,h,1));for(b=0;b<t.j.g;b++)t.e[b]&&++t.c;for(t.d=j(l.Feb,ht,6,t.j.f,16,1),t.f=j(l.Feb,ht,6,t.j.f,16,1),u=0;u<t.j.f;u++){for(c=0,g=0;g<e.$o(t.j,u);g++)t.e[e._o(t.j,u,g)]&&++c;c>0&&(t.d[u]=true,++t.b,c==3&&(t.f[u]=true));}for(t.i=j(l.Feb,ht,6,n.j.a.length,16,1),k=0;k<n.j.a.length;k++)for(t.i[k]=true,Q=y.xj(n.k,k),L=0,s=Q.length;L<s;++L)if(o=Q[L],!t.e[o]){t.i[k]=false;break;}for(t.g=j(l.Feb,ht,6,t.j.f,16,1),B=0;B<n.j.a.length;B++)if(t.i[B]&&y.xj(n.k,B).length!=6)for(r=y.xj(n.j,B),i=0,D=r.length;i<D;++i)P=r[i],y.xj(n.k,B).length==5&&(t.g[P]=true);t.k=0;},e.Fe=function(t,A,n){var P,r;for(r=0;r<e.$o(t.j,A);r++)if(P=e.Zo(t.j,A,r),e.Tl(t.j,P)==8&&e.$o(t.j,P)==1)return n&&e.Bl(t.j,P)==0&&e.jn(t.j,P,-1),true;return  false;},e.Ge=function(t,A,n){var P,r,i,D;if(e.bw(t.j,1),D=new e.Dt(t.j,1),e.Ee(t,A,D),t.c==0)return  true;if(n)for(r=0;r<t.j.f;r++)t.d[r]&&e.Tl(t.j,r)==7&&(t.f[r]||e.$o(t.j,r)==3&&!t.g[r])&&e.jn(t.j,r,1);for(t.a=false,e.Re(t,n),t.j.K&&e.Je(t),e.Te(t,D),e.Oe(t,D),e.Qe(t),e.Ke(t);e.Me(t,D,n);)e.Ke(t);for(;e.Ne(t,D,n);)e.Ke(t);for(;e.Le(t,D,n);)e.Ke(t);for(;t.c!=0;)for(i=0;i<t.j.g;i++)t.e[i]&&(e.Ie(t,i),e.Ke(t));for(n&&e.Ue(t,n),t.b-t.k>=2&&e.Ce(t),P=0;P<t.j.f;P++)t.d[P]&&e.op(t.j,P)!=0&&(n&&(e.Bl(t.j,P)==1&&e.Hm(t.j,P)||e.Bl(t.j,P)==-1&&e.Tl(t.j,P)==5)?e.jn(t.j,P,0):e.wn(t.j,P,32),++t.k);return t.b==t.k;},e.He=function(t,A){return t==A-1?0:t+1;},e.Ie=function(t,A){var n,P,r,i;for(e.im(t.j,A)==1&&(e.Mn(t.j,A,2),t.k+=2),r=0;r<2;r++)for(n=e.Zl(t.j,r,A),t.d[n]=false,i=0;i<e.$o(t.j,n);i++)P=e._o(t.j,n,i),t.e[P]&&(t.e[P]=false,--t.c);},e.Je=function(t){var A,n,P,r,i,D,u,o;for(n=0;n<t.j.g;n++)if(t.e[n])for(r=0;r<2;r++){for(u=e.Zl(t.j,r,n),A=false,D=0;D<e.$o(t.j,u);D++)if(n!=e._o(t.j,u,D)&&t.e[e._o(t.j,u,D)]){A=true;break;}if(!A){for(o=n,P=e.Zl(t.j,1-r,n);o!=-1;)for(t.d[u]=false,t.e[o]=false,--t.c,e.Mn(t.j,o,8),o=-1,u=P,i=0;i<e.$o(t.j,u);i++)if(t.e[e._o(t.j,u,i)])if(o==-1)o=e._o(t.j,u,i),P=e.Zo(t.j,u,i);else {u=-1,o=-1;break;}u!=-1&&(t.d[P]=false);break;}}},e.Ke=function(t){var A,n,P,r,i,D,u;do for(u=false,n=0;n<t.j.g;n++)if(t.e[n]){for(i=false,r=0;r<2;r++){for(P=e.Zl(t.j,r,n),A=false,D=0;D<e.$o(t.j,P);D++)if(n!=e._o(t.j,P,D)&&t.e[e._o(t.j,P,D)]){A=true;break;}if(!A){i=true;break;}}i&&(u=true,e.Ie(t,n));}while(u);},e.Le=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(c=0;c<A.j.a.length;c++)if(y.xj(A.k,c).length!=6&&t.i[c]){for(s=true,B=y.xj(A.k,c),L=0;L<B.length;L++)if(!t.e[B[L]]){s=false;break;}if(s){for(u=-1,o=0,g=y.xj(A.j,c),Q=0;Q<g.length;Q++)P=g[Q],b=t.g[P]?e.ye(t,P,false):e.ze(t,P,false),o<b&&(o=b,u=Q);if(u!=-1){if(h=g[u],n){for(r=g,i=0,D=r.length;i<D;++i)P=r[i],P!=h&&e.Be(t,P,true);t.g[h]?e.ye(t,h,true):e.ze(t,h,true);}return e.Pe(t,h),t.g[h]&&e.Se(t,g,B,u),true;}}}return  false;},e.Me=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;for(M=j(l.j7,H,6,t.j.g,15,1),g=0;g<A.j.a.length;g++){for(k=y.xj(A.k,g),s=true,o=0;o<k.length;o++)if(!t.e[k[o]]){s=false;break;}if(s)for(u=0;u<k.length;u++)++M[k[u]];}for(r=t.c,c=0;c<A.j.a.length;c++)for(D=false,B=y.xj(A.j,c),k=y.xj(A.k,c),u=0;u<k.length&&!D;u++)if(M[k[u]]>1&&(i=e.He(u,k.length),M[k[i]]==1)){for(Q=false,b=e.He(i,k.length);M[k[b]]==1;)(!e.Be(t,B[b],false)||k.length!=6&&e.Tl(t.j,B[b])!=6)&&(Q=true),b=e.He(b,k.length);if(!Q&&(P=b>i?b-i:b+k.length-i,P>2&&(P&1)==1)){for(h=1;h<P;h+=2)L=i+h<k.length?i+h:i+h-k.length,n&&(e.Be(t,B[L],true),e.Be(t,B[L==B.length-1?0:L+1],true)),e.Ie(t,k[L]);D=true;}}return r!=t.c;},e.Ne=function(t,A,n){var P,r,i,D,u,o,Q,L;for(P=false,o=0;o<A.j.a.length;o++)if(y.xj(A.k,o).length==6){for(u=true,Q=y.xj(A.j,o),L=y.xj(A.k,o),i=0;i<6;i++)if(!e.Be(t,Q[i],false)||!t.e[L[i]]){u=false;break;}if(u){if(n)for(D=0;D<6;D++)e.Be(t,Q[D],true);for(r=0;r<6;r+=2)e.Ie(t,L[r]);P=true;break;}}return P;},e.Oe=function(t,A){var n;for(n=0;n<t.j.g;n++)t.e[n]&&e.wt(A,n)&&(e.Pe(t,e.Zl(t.j,0,n)),e.Pe(t,e.Zl(t.j,1,n)));},e.Pe=function(t,A){var n,P;for(t.d[A]&&(t.d[A]=false,--t.b),P=0;P<e.$o(t.j,A);P++)n=e._o(t.j,A,P),t.e[n]&&(t.e[n]=false,--t.c);},e.Qe=function(t){var A,n,P,r,i;for(n=0;n<t.j.g;n++)if(e.fm(t.j,n)==2){for(r=0;r<2;r++)if(A=e.Zl(t.j,r,n),e.Tl(t.j,A)<=8){for(i=0;i<e.$o(t.j,A);i++)if(P=e._o(t.j,A,i),t.e[P]){e.Pe(t,A);break;}}}},e.Re=function(t,A){var n;for(n=0;n<t.j.f;n++)t.d[n]&&e.pp(t.j,n)==0&&(!A||e.Tl(t.j,n)==5&&e.Bl(t.j,n)<0||e.Tl(t.j,n)==6||e.Tl(t.j,n)==14||e.Hm(t.j,n)&&e.Bl(t.j,n)>0)&&e.Pe(t,n);},e.Se=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B;if(s=A[P==0?4:P-1],B=A[P==4?0:P+1],e.$o(t.j,s)==3&&e.$o(t.j,B)==3){for(b=j(l.j7,H,6,t.j.f,15,1),h=j(l.Feb,ht,6,t.j.g,16,1),i=n,D=0,u=i.length;D<u;++D)o=i[D],h[o]=true;for(r=0;r<t.j.g;r++)t.e[r]||(h[r]=true);if(c=e.yp(t.j,b,s,B,14,null,h),c==14){for(g=true,L=2;L<=12;L++)if((L&3)!=1^e.Uo(t.j,b[L])==5){g=false;break;}if(g)for(e.Pe(t,b[7]),Q=1;Q<=5;Q+=2)e.Ie(t,e.Xo(t.j,b[Q],b[Q+1])),e.Ie(t,e.Xo(t.j,b[Q+7],b[Q+8]));}}},e.Te=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G;for(p=j(l.Feb,ht,6,t.j.f,16,1),z=0;z<A.j.a.length;z++)if(tt=y.xj(A.k,z).length,(tt==3||tt==5||tt==6||tt==7)&&t.i[z])for(P=y.xj(A.j,z),r=0,i=P.length;r<i;++r)n=P[r],p[n]=true;for(h=j(l.j7,H,6,t.j.f,15,1),c=j(l.Feb,ht,6,t.j.f,16,1),Q=0;Q<t.j.g;Q++)D=e.Zl(t.j,0,Q),u=e.Zl(t.j,1,Q),!p[D]&&!p[u]&&(t.e[Q]&&(++h[D],++h[u]),e.im(t.j,Q)==16&&(c[D]=true,c[u]=true));for(Z=j(l.j7,H,6,t.j.f,15,1),b=j(l.j7,H,6,t.j.f,15,1),G=0;G<t.j.f;G++)if(h[G]==1)for(b[0]=G,s=0,g=0;s<=g;){for(B=0;B<e.$o(t.j,b[s]);B++)if(t.e[e._o(t.j,b[s],B)]&&(L=e.Zo(t.j,b[s],B),(s==0||L!=b[s-1])&&h[L]!=0)){if(b[++g]=L,(h[L]&1)!=0){for(S=1;S<g;S+=2)Z[b[S]]=-1;g=0;}break;}++s;}for(o=j(l.Feb,ht,6,t.j.f,16,1),ot=0;ot<t.j.f;ot++)if(!o[ot]&&h[ot]!=0){for(b[0]=ot,o[ot]=true,s=0,g=0;s<=g;){for(B=0;B<e.$o(t.j,b[s]);B++)t.e[e._o(t.j,b[s],B)]&&(L=e.Zo(t.j,b[s],B),o[L]||(b[++g]=L,o[L]=true));++s;}if((g&1)==0){for(k=0;k<=g;k++)Z[b[k]]==0&&(Z[b[k]]=e.Ae(t,b[k],false));for(q=true,M=0;M<=g;M++)if(Z[b[M]]<=0&&!e.Be(t,b[M],false)){q=false;break;}if(q){for(I=0,V=-1,B=0;B<=g;B++)I<Z[b[B]]&&(I=Z[b[B]],V=b[B]);I>0&&(e.Ae(t,V,true),e.Pe(t,V));}}}},e.Ue=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g;for(n=0;n<t.j.f;n++)if(e.bw(t.j,1),e.Tl(t.j,n)==7&&e.Bl(t.j,n)==1&&e.Qo(t.j,n)==1&&!e.Fe(t,n,A))for(u=false,o=j(l.j7,H,6,t.j.f,15,1),g=j(l.j7,H,6,t.j.f,15,1),Q=j(l.j7,H,6,t.j.f,15,1),o[0]=n,g[n]=-1,Q[n]=1,i=0,L=0;i<=L&&!u;){for(D=o[i],s=0;s<e.$o(t.j,D)&&!u;s++)if(h=(Q[D]&1)==0^e.fm(t.j,e._o(t.j,D,s))>1,r=e.Zo(t.j,D,s),Q[r]==0&&h)if(e.Tl(t.j,r)==7&&e.Qo(t.j,r)==0&&e.$o(t.j,r)==2){if((Q[D]&1)==0){for(e.jn(t.j,n,0),c=D,b=0;b<Q[D];b++)P=e.Xo(t.j,r,c),e.In(t.j,P,e.fm(t.j,P)==1?2:e.fm(t.j,P)-1),r=c,c=g[r];u=true;}}else o[++L]=r,g[r]=D,Q[r]=Q[D]+1;++i;}},e.Ve=function(t){e.We.call(this,t);},e.We=function(t){this.j=t;},N(89,1,{},e.Ve),U.a=false,U.b=0,U.c=0,U.k=0,l.q7=O(89),e.Xe=function(t,A){var n,P;for(n=0,P=0;P<t.j[A];P++)t.n[A][P]==2&&(e.Tl(t,t.i[A][P])==7||e.Tl(t,t.i[A][P])==8||e.Tl(t,t.i[A][P])==16)&&++n;return n;},e.Ye=function(t,A){var n,P,r,i,D,u;if(t.s[A]==0)return  false;for(u=true,n=t.s[A],i=t.j[A],D=0,P=0;P<i;P++)r=t.i[A][P],D+=t.s[r];return d.Math.abs(n)<=d.Math.abs(D)&&T.Tib(n)!=T.Tib(D)&&(u=false),u;},e.Ze=function(t,A){var n,P,r,i,D,u,o;if(t.C[A]!=8||t.s[A]!=0||t.j[A]!=1||t.n[A][0]!=1)return  false;if(n=t.i[A][0],t.C[n]==6){for(D=t.j[n],P=0;P<D;P++)if(r=t.i[n][P],r!=A&&t.C[r]==8&&(i=e.Xo(t,n,r),t.H[i]==2))return  true;}else if(t.C[n]==7){if(t.s[n]==1)return  true;}else if(t.C[n]==16){for(u=t.j[n],o=0,P=0;P<u;P++)r=t.i[n][P],r!=A&&t.C[r]==8&&(i=e.Xo(t,n,r),t.H[i]==2&&++o);if(o==2)return  true;}else if(e.$e(t,A))return  true;return  false;},e.$e=function(t,A){var n,P,r,i,D;if(t.C[A]!=8||t.j[A]!=1)return  false;if(n=t.i[A][0],t.C[n]==15){for(D=t.j[n],P=0;P<D;P++)if(r=t.i[n][P],r!=A&&t.C[r]==8&&(i=e.Xo(t,n,r),t.H[i]==2))return  true;}return  false;},e._e=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I;if(t.C[A]!=7||t.s[A]!=0||t.j[A]+t.o[A]>3)return  false;if(A<t.f&&e.ot(t.p,A)){if(t.o[A]!=1||e.To(t,A,7)!=1)return  false;for(q=(e.bw(t,7),t.p),M=0;M<q.j.a.length;M++)if(e.qt(q,M,A)){if(y.xj(q.k,M).length==5||y.xj(q.k,M).length==6){for(S=y.xj(q.j,M),B=-1,o=0;o<S.length;o++)if(S[o]==A){B=o;break;}for(r=0,k=null,g=null,S.length==5&&(k=j(l.j7,H,6,2,15,1),k[0]=S[B-1<0?B+4:B-1],k[1]=S[B-4<0?B+1:B-4],g=j(l.j7,H,6,2,15,1),g[0]=S[B-2<0?B+3:B-2],g[1]=S[B-3<0?B+2:B-3]),S.length==6&&(k=j(l.j7,H,6,3,15,1),k[0]=S[B-1<0?B+5:B-1],k[1]=S[B-3<0?B+3:B-3],k[2]=S[B-5<0?B+1:B-5],g=j(l.j7,H,6,2,15,1),g[0]=S[B-2<0?B+4:B-2],g[1]=S[B-4<0?B+2:B-4]),Q=0;Q<S.length;Q++)A!=S[Q]&&e.Tl(t,S[Q])==7&&e.Qo(t,S[Q])==1&&--r;for(L=0;L<k.length;L++){for(i=-1,D=-1,c=0;c<e.$o(t,k[L]);c++)if(!e.Up(t,e._o(t,k[L],c))){i=e.Zo(t,k[L],c),D=e._o(t,k[L],c);break;}if(i!=-1){if(t.C[i]==7&&t.o[i]==0&&t.j[i]+t.o[i]<=3&&!e.af(t,i,false)){++r;continue;}if(t.C[i]==8&&t.j[i]==1){r+=2;continue;}if(D<t.g&&e.pt(t.p,D)){for(V=0;V<q.j.a.length;V++)if(q.e[V]&&e.qt(q,V,i)){for(p=y.xj(q.j,V),b=0;b<p.length;b++)if(e.Tl(t,p[b])==7&&e.Qo(t,p[b])==1){--r;break;}break;}}}}for(s=0;s<g.length;s++){for(i=-1,b=0;b<e.$o(t,g[s]);b++)e.Up(t,e._o(t,g[s],b))||(i=e.Zo(t,g[s],b));e.Tl(t,g[s])==6?i!=-1&&e.Xe(t,i)!=0&&--r:e.Tl(t,g[s])==7&&e.Qo(t,g[s])==0&&(i==-1||!(i<t.f&&e.ot(t.p,i))&&e.Xe(t,i)==0)&&++r;}return r>0;}break;}return  false;}if(t.o[A]>1)return  false;if(t.o[A]==1){for(h=-1,I=0,o=0;o<t.j[A];o++){if(P=t.i[A][o],t.n[A][o]==2){if(t.C[P]!=6)return  false;h=P;continue;}if(t.C[P]==8)return  false;if(t.C[P]==7){--I,e.af(t,P,false)&&--I;continue;}P<t.f&&e.ot(t.p,P)&&--I;}if(h==-1)return  false;for(n=0,Q=0;Q<t.j[h];Q++)if(t.n[h][Q]==1){if(P=t.i[h][Q],e.Xe(t,P)!=0)return  false;P<t.f&&e.ot(t.p,P)&&++n,t.C[P]==7&&!e.af(t,P,true)&&++I,(t.C[P]==8||t.C[P]==16)&&--I;}return n==2&&--I,I>=0;}for(u=0;u<t.j[A];u++)if(P=t.i[A][u],P<t.f&&e.ot(t.p,P)||t.C[P]!=6||e.Xe(t,P)!=0||t.o[P]!=0&&e.bf(t,P))return  false;return  true;},e.af=function(t,A,n){var P,r,i;for(P=false,i=0;i<t.j[A];i++)if(!e.Up(t,t.k[A][i])&&t.n[A][i]==1&&(r=t.i[A][i],!(r<t.f&&e.ot(t.p,r))&&(t.C[r]==6&&e.Xe(t,r)==1||t.C[r]==16&&e.Xe(t,r)==2))){if(P||!n)return  true;P=true;}return  false;},e.bf=function(t,A){var n,P,r;for(P=0;P<t.j[A];P++)if(t.n[A][P]!=1){for(n=t.i[A][P],r=0;r<t.j[n];r++)if(t.n[n][r]==1&&e.Xe(t,t.i[n][r])!=0)return  true;}return  false;},e.ef=function(){e.ef=D1,e.cf=v(F(l.Eeb,1),Yt,6,15,[-1,-1,-1,0,0,1,2,3,4,5,-1,0,0,0,6,7,8,9,-1,0,0,10,10,10,10,10,10,10,10,10,10,1,11,11,12,13,-1,0,0,10,10,10,10,10,10,10,10,10,10,0,0,0,11,14,-1,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,10,10,10,10,10,10,10,10,1,1,1,1,-1,-1,-1,-1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]),e.df=v(F(l.Eeb,1),Yt,6,15,[-1,-1,-1,0,0,0,2,5,5,5,-1,0,0,0,0,9,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);},e.ff=function(t,A,n){e.ef();var P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(e.bw(t,7),b=j(l.k7,e0,6,t.j[A],14,1),h=0,o=0;o<t.j[A];o++)if(r=t.i[A][o],t.C[r]!=1){if(D=0,(n&32)!=0&&(i=Mt(t.n[A][o]),n==32190?et(i,3)<0&&e.Yp(t,t.k[A][o])&&t.o[A]==1&&(i=0):et(i,3)<0&&e.Up(t,t.k[A][o])&&(i=0),D=rt(D,h1(i,4))),(n&128)!=0){if(e.cf[t.C[r]]==-1)throw Ut(new T.cC(fr+t.C[r]));D=m1(D,Mt(e.cf[t.C[r]]));}else if((n&64)!=0){if(e.df[t.C[r]]==-1)throw Ut(new T.cC(fr+t.C[r]));D=m1(D,Mt(e.df[t.C[r]]));}for((n&256)!=0&&(c=t.j[r]-1,c>3&&(c=3),(n&512)==0&&c>1&&(c=1),D=rt(D,l.Ueb(c<<6))),(n&i0)!=0&&(t.u[r]&8)!=0&&(D=rt(D,256)),(n&j1)!=0&&r<t.f&&e.ot(t.p,r)&&(D=rt(D,512)),(n&I0)!=0&&(t.u[r]&L0)!=0&&(D=rt(D,i0)),L=0;et(D,b[L])<0;)++L;for(s=o;s>L;s--)b[s]=b[s-1];b[L]=D,++h;}for(h>4&&(h=4),P=0,Q=0;Q<h;Q++)P=h1(P,11),P=rt(P,b[Q]);if(P=h1(P,15),e.cf[t.C[A]]==-1)throw Ut(new T.cC(fr+t.C[A]));if(P=rt(P,Mt(e.cf[t.C[A]])),(n&2)!=0?(B=t.p&&A<t.f?e.dt(t.p,A):0,B>9&&(B=9),B>2&&(B-=2),P=rt(P,l.Ueb(B<<4))):(n&1)!=0&&(t.u[A]&8)!=0&&(P=rt(P,64)),(n&4)!=0&&A<t.f&&e.ot(t.p,A)&&(P=rt(P,i0)),(n&8)!=0&&(t.u[A]&Z1)!=0&&(P=rt(P,j1)),(n&16)!=0&&(t.u[A]&L0)!=0&&(P=rt(P,Z1)),(n&Z1)!=0&&(e.Ye(t,A)&&(P=rt(P,L0)),e._e(t,A))){for(u=0;u<t.f;u++)if(e.Ze(t,u)){P=rt(P,I0);break;}}return (n&L0)!=0&&(g=Mt(e.To(t,A,10)),P=rt(P,h1(g,7))),P;},e.gf=function(t,A,n,P,r,i){var D,u,o,Q;for(o=0,u=0;u<t.R.f;u++)pt(J(e.Ol(t.R,t.B[u]),P),0)&&++o;if(o!=0)for(e.eg(t,A),e._f(t,l.Ueb(o),n),D=0;D<t.R.f;D++)Q=J(e.Ol(t.R,t.B[D]),P),et(Q,0)!=0&&(e._f(t,l.Ueb(D),n),r!=1&&e._f(t,t0(Q,i),r));},e.hf=function(t,A,n,P,r,i){var D,u,o,Q;for(o=0,u=0;u<t.R.g;u++)(e.hm(t.R,t.C[u])&P)!=0&&++o;if(o!=0)for(e.eg(t,A),e._f(t,l.Ueb(o),n),D=0;D<t.R.g;D++)Q=e.hm(t.R,t.C[D])&P,Q!=0&&(e._f(t,l.Ueb(D),n),r!=1&&e._f(t,l.Ueb(Q>>i),r));},e.jf=function(t){for(var A,n,P,r,i,D;;){for(i=j(l.Feb,ht,6,t.U+1,16,1),P=-1,n=0;n<t.R.f;n++)e.pp(t.R,n)!=0&&(i[t.f[n]]&&P<t.f[n]&&(P=t.f[n]),i[t.f[n]]=true);if(P==-1)break;for(r=0,A=0;A<t.R.f;A++)D=0,t.f[A]==P&&(D=++r),e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),e.ah(t.e[A],8,l.Ueb(D));t.U=e.Kf(t);}},e.kf=function(t){var A,n,P,r;for(n=0;n<t.R.f;n++)e.eh(t.e[n],n),e.ah(t.e[n],2*t.c+4,h1(Mt(t.f[n]),t.c+4));for(r=false,A=0;A<t.R.f;A++)r=r|e.xf(t,A,3);for(P=0;P<t.R.g;P++)r=r|e.sf(t,P,3);if(!r)return  false;for(;t.U<t.R.f&&(r=e.Hf(t),!!r);)e.Jf(t),t.P&&e.uh(t.P,t.f);return  true;},e.lf=function(t){var A,n,P,r,i;for(n=0;n<t.R.f;n++)e.eh(t.e[n],n),e.ah(t.e[n],t.c+1,z0(2,Mt(t.f[n])));for(i=j(l.j7,H,6,t.U+1,15,1),P=0;P<t.R.f;P++)++i[t.f[P]];for(r=1;i[r]==1;)++r;for(A=0;A<t.R.f;A++)if(t.f[A]==r){e.bh(t.e[A],1);break;}t.U=e.Kf(t);},e.mf=function(t,A,n){var P,r,i,D,u;return e.Tl(t.R,A)!=6&&e.Tl(t.R,A)!=7||(r=e.Zo(t.R,A,0),i=e.Zo(t.R,A,1),e.Qo(t.R,r)!=1||e.Qo(t.R,i)!=1)||e.$o(t.R,r)==1||e.$o(t.R,i)==1||e.Lo(t.R,r)>3||e.Lo(t.R,i)>3||(D=new e.lk(t.R,t.f,A,r),D.f&&n==1)||(u=new e.lk(t.R,t.f,A,i),u.f&&n==1)||D.f&&u.f?false:(n==3&&(D.f&&D.c&&(t.W[A]=true),u.f&&u.c&&(t.W[A]=true)),P=t.jb?e.of(t,D,u):e.nf(D,u),n==1?t.db[A]=P:n==2&&(D.f&&(P==1?e.bh(t.e[D.b],Mt(t.f[r])):e.bh(t.e[D.d],Mt(t.f[r]))),u.f&&(P==2?e.bh(t.e[u.b],Mt(t.f[i])):e.bh(t.e[u.d],Mt(t.f[i])))),true);},e.nf=function(t,A){var n,P,r;if(P=e.kk(t),r=e.kk(A),P==-1||r==-1||(P+r&1)==0)return 3;switch(n=0,P+r){case 3:case 7:n=2;break;case 5:n=1;}return n;},e.of=function(t,A,n){var P,r;return P=j(l.j7,H,6,4,15,1),P[0]=A.b,P[1]=A.a,P[2]=n.a,P[3]=n.b,r=e.el(t.R,P),d.Math.abs(r)<.3||d.Math.abs(r)>O8?3:r<0?2:1;},e.pf=function(t,A,n){var P,r,i,D,u;return !e.Vp(t.R,A)||(P=e.Zl(t.R,0,A),r=e.Zl(t.R,1,A),D=new e.lk(t.R,t.f,P,r),D.f&&n==1)||(u=new e.lk(t.R,t.f,r,P),u.f&&n==1)||D.f&&u.f?false:(n==3&&(D.f&&(t.V[A]=e.xg(t,r)),u.f&&(t.V[A]=e.xg(t,P))),i=t.jb?e.rf(t,D,u):e.qf(D,u),n==1?t.o[A]=i:n==2&&(D.f&&(i==2?e.bh(t.e[D.b],Mt(t.f[r])):e.bh(t.e[D.d],Mt(t.f[r]))),u.f&&(i==2?e.bh(t.e[u.b],Mt(t.f[P])):e.bh(t.e[u.d],Mt(t.f[P])))),true);},e.qf=function(t,A){var n,P,r;if(P=e.kk(t),r=e.kk(A),P==-1||r==-1||(P+r&1)==0)return 3;switch(n=0,P+r){case 3:case 7:n=1;break;case 5:n=2;}return n;},e.rf=function(t,A,n){var P,r;return P=j(l.j7,H,6,4,15,1),P[0]=A.b,P[1]=A.a,P[2]=n.a,P[3]=n.b,r=e.el(t.R,P),d.Math.abs(r)<.3||d.Math.abs(r)>O8?3:r<0?1:2;},e.sf=function(t,A,n){var P,r,i,D,u,o,Q,L;if(t.o[A]!=0)return  false;if(t.R.K){for(Q=0;Q<2;Q++)for(P=e.Zl(t.R,Q,A),L=0;L<e.Lo(t.R,P);L++)if(e.Jm(t.R,e.Zo(t.R,P,L)))return  false;}return e.fm(t.R,A)==1?e.pf(t,A,n):e.fm(t.R,A)!=2||e.Up(t.R,A)||(i=e.Zl(t.R,0,A),D=e.Zl(t.R,1,A),e.$o(t.R,i)==1||e.$o(t.R,D)==1)||e.$o(t.R,i)>3||e.$o(t.R,D)>3||e.Qo(t.R,i)==2||e.Qo(t.R,D)==2||(u=new e.lk(t.R,t.f,D,i),u.f&&n==1)||(o=new e.lk(t.R,t.f,i,D),o.f&&n==1)||u.f&&o.f?false:(n==3&&(u.f&&u.c&&(t.V[A]=true),o.f&&o.c&&(t.V[A]=true)),r=e.Gm(t.R,A)?3:t.jb?e.uf(t,u,o):e.tf(u,o),n==1?t.o[A]=r:n==2&&(u.f&&(r==1?e.bh(t.e[u.b],Mt(t.f[i])):r==2&&e.bh(t.e[u.d],Mt(t.f[i]))),o.f&&(r==1?e.bh(t.e[o.b],Mt(t.f[D])):r==2&&e.bh(t.e[o.d],Mt(t.f[D])))),true);},e.tf=function(t,A){return e.kk(t)==-1||e.kk(A)==-1||((e.kk(t)|e.kk(A))&1)!=0?3:e.kk(t)==e.kk(A)?1:2;},e.uf=function(t,A,n){var P,r,i,D,u,o,Q;return i=j(l.h7,Ft,6,3,15,1),i[0]=e.Ql(t.R,n.a)-e.Ql(t.R,A.a),i[1]=e.Rl(t.R,n.a)-e.Rl(t.R,A.a),i[2]=e.Sl(t.R,n.a)-e.Sl(t.R,A.a),o=j(l.h7,Ft,6,3,15,1),o[0]=e.Ql(t.R,A.b)-e.Ql(t.R,A.a),o[1]=e.Rl(t.R,A.b)-e.Rl(t.R,A.a),o[2]=e.Sl(t.R,A.b)-e.Sl(t.R,A.a),Q=j(l.h7,Ft,6,3,15,1),Q[0]=e.Ql(t.R,n.b)-e.Ql(t.R,n.a),Q[1]=e.Rl(t.R,n.b)-e.Rl(t.R,n.a),Q[2]=e.Sl(t.R,n.b)-e.Sl(t.R,n.a),D=j(l.h7,Ft,6,3,15,1),D[0]=i[1]*o[2]-i[2]*o[1],D[1]=i[2]*o[0]-i[0]*o[2],D[2]=i[0]*o[1]-i[1]*o[0],u=j(l.h7,Ft,6,3,15,1),u[0]=i[1]*D[2]-i[2]*D[1],u[1]=i[2]*D[0]-i[0]*D[2],u[2]=i[0]*D[1]-i[1]*D[0],P=(o[0]*u[0]+o[1]*u[1]+o[2]*u[2])/(d.Math.sqrt(o[0]*o[0]+o[1]*o[1]+o[2]*o[2])*d.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),r=(Q[0]*u[0]+Q[1]*u[1]+Q[2]*u[2])/(d.Math.sqrt(Q[0]*Q[0]+Q[1]*Q[1]+Q[2]*Q[2])*d.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),P<0^r<0?1:2;},e.vf=function(t,A){var n,P,r,i;return n=e.zl(t.R,A),P=e.mp(t.R,A,false),r=e.mp(t.R,A,true),i=-1,P!=r?n!=-1&&n>P?i=n<<24>>24:i=P<<24>>24:n!=-1?(n>r||n<r&&n>=e.vp(t.R,A))&&(i=n<<24>>24):!e.Aq(t.R,A)&&e.cp(t.R,A)!=0&&(i=e.vp(t.R,A),i-=e.mm(t.R,A,i)),e.Rf(t,A,i),i;},e.wf=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(P=j(l.j7,H,6,t.O,15,1),A=0;A<t.R.f;A++){for(L=e.$o(t.R,A)+e.rp(t.R,A),Q=0,i=0;i<e.Mo(t.R,A);i++)if(i<e.$o(t.R,A)||i>=e.Lo(t.R,A)){for(s=2*t.f[e.Zo(t.R,A,i)],n=e._o(t.R,A,i),e.fm(t.R,n)==2&&(e.Up(t.R,n)||++s),u=0;u<Q&&!(s<P[u]);u++);for(o=Q;o>u;o--)P[o]=P[o-1];P[u]=s,++Q;}for(e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),D=L;D<t.O;D++)e.ah(t.e[A],t.c+1,0);for(r=0;r<L;r++)e.ah(t.e[A],t.c+1,l.Ueb(P[r]));}},e.xf=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g;if(t.db[A]!=0||e.Tl(t.R,A)!=5&&e.Tl(t.R,A)!=6&&e.Tl(t.R,A)!=7&&e.Tl(t.R,A)!=14&&e.Tl(t.R,A)!=15&&e.Tl(t.R,A)!=16||(t.Q&Z1)!=0&&e.Tl(t.R,A)>=14)return  false;if(e.Qo(t.R,A)!=0){if(e.Wp(t.R,A))return e.mf(t,A,n);if(e.Tl(t.R,A)!=15&&e.Tl(t.R,A)!=16)return  false;}if(e.$o(t.R,A)<3||e.Lo(t.R,A)>4||e.Bl(t.R,A)>0&&e.Tl(t.R,A)==6||e.Tl(t.R,A)==5&&e.Lo(t.R,A)!=4)return  false;if(t.R.K){if(e.Jm(t.R,A))return  false;for(u=0;u<e.Lo(t.R,A);u++)if(e.Jm(t.R,e.Zo(t.R,A,u)))return  false;}if(e.Tl(t.R,A)==7&&!t.S[A])return  false;for(c=j(l.j7,H,6,4,15,1),g=j(l.j7,H,6,4,15,1),L=j(l.Feb,ht,6,4,16,1),o=0;o<e.Lo(t.R,A);o++){for(i=-1,r=0,Q=0;Q<e.Lo(t.R,A);Q++)L[Q]||i<t.f[e.Zo(t.R,A,Q)]&&(i=t.f[e.Zo(t.R,A,Q)],r=Q);c[o]=r,g[o]=i,L[r]=true;}if(e.Lo(t.R,A)==4&&g[0]==g[1]&&g[2]==g[3]||e.Lo(t.R,A)==4&&(g[0]==g[2]||g[1]==g[3])||e.Lo(t.R,A)==3&&g[0]==g[2])return  false;for(s=0,h=0,b=false,D=1;D<e.Lo(t.R,A);D++)if(g[D-1]==g[D]){if(n==1||g[D]==0)return  false;s=e.Zo(t.R,A,c[D-1]),h=e.Zo(t.R,A,c[D]),n==3&&e.gq(t.R,e._o(t.R,A,c[D]))&&(t.W[A]=true),b=true;}return n!=1&&!b?false:(P=t.jb?e.zf(t,A,c):e.yf(t,A,c),n==1?t.db[A]=P:n==2&&(P==1?e.bh(t.e[s],Mt(t.f[A])):P==2&&e.bh(t.e[h],Mt(t.f[A]))),true);},e.yf=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h;for(h=v(F(l.j7,2),E1,7,0,[v(F(l.j7,1),H,6,15,[2,1,2,1]),v(F(l.j7,1),H,6,15,[1,2,2,1]),v(F(l.j7,1),H,6,15,[1,1,2,2]),v(F(l.j7,1),H,6,15,[2,1,1,2]),v(F(l.j7,1),H,6,15,[2,2,1,1]),v(F(l.j7,1),H,6,15,[1,2,1,2])]),P=j(l.h7,Ft,6,e.Lo(t.R,A),15,1),D=0;D<e.Lo(t.R,A);D++)P[D]=e.Yl(t.R,e.Zo(t.R,A,n[D]),A);if(Q=e.ep(t.R,A,n,P,null)<<24>>24,Q!=3)return Q;for(L=0,s=0,u=0;u<e.Lo(t.R,A);u++)r=e._o(t.R,A,n[u]),e.Zl(t.R,0,r)==A&&(e.im(t.R,r)==129&&(s!=0&&e.Vn(t.R,A),L=u,s=1),e.im(t.R,r)==257&&(s!=0&&e.Vn(t.R,A),L=u,s=2));if(s==0)return 3;for(i=1;i<e.Lo(t.R,A);i++)P[i]<P[0]&&(P[i]+=I1);if(e.Lo(t.R,A)==3){switch(L){case 0:(P[1]<P[2]&&P[2]-P[1]<A1||P[1]>P[2]&&P[1]-P[2]>A1)&&(s=3-s);break;case 1:P[2]-P[0]>A1&&(s=3-s);break;case 2:P[1]-P[0]<A1&&(s=3-s);}return s==1?2:1;}return o=0,P[1]<=P[2]&&P[2]<=P[3]?o=0:P[1]<=P[3]&&P[3]<=P[2]?o=1:P[2]<=P[1]&&P[1]<=P[3]?o=2:P[2]<=P[3]&&P[3]<=P[1]?o=3:P[3]<=P[1]&&P[1]<=P[2]?o=4:P[3]<=P[2]&&P[2]<=P[1]&&(o=5),h[o][L]==s?2:1;},e.zf=function(t,A,n){var P,r,i,D,u,o;for(P=j(l.j7,H,6,4,15,1),u=0;u<e.Lo(t.R,A);u++)P[u]=e.Zo(t.R,A,n[u]);for(e.Lo(t.R,A)==3&&(P[3]=A),r=y1(l.h7,[i1,Ft],[15,6],15,[3,3],2),D=0;D<3;D++)r[D][0]=e.Ql(t.R,P[D+1])-e.Ql(t.R,P[0]),r[D][1]=e.Rl(t.R,P[D+1])-e.Rl(t.R,P[0]),r[D][2]=e.Sl(t.R,P[D+1])-e.Sl(t.R,P[0]);return o=j(l.h7,Ft,6,3,15,1),o[0]=r[0][1]*r[1][2]-r[0][2]*r[1][1],o[1]=r[0][2]*r[1][0]-r[0][0]*r[1][2],o[2]=r[0][0]*r[1][1]-r[0][1]*r[1][0],i=(r[2][0]*o[0]+r[2][1]*o[1]+r[2][2]*o[2])/(d.Math.sqrt(r[2][0]*r[2][0]+r[2][1]*r[2][1]+r[2][2]*r[2][2])*d.Math.sqrt(o[0]*o[0]+o[1]*o[1]+o[2]*o[2])),i>0?1:2;},e.Af=function(t){var A,n;for(A=0,y.brb(t.e),n=0;n<t.e.length;n++)(n==0||e.dh(t.e[n],t.e[n-1])!=0)&&++A,t.f[t.e[n].a]=A;return A;},e.Bf=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h;if(!t.A){for(t.A=new y.Mj(),L=0,s=j(l.j7,H,6,t.R.f,15,1),D=j(l.j7,H,6,t.R.f,15,1),o=j(l.j7,H,6,t.R.g,15,1),A=0;A<t.R.f;A++)if(s[A]==0&&(e.fq(t.R,A)||e.Qo(t.R,A)==1)){for(D[0]=A,u=1,Q=0,s[A]=++L,n=j(l.Feb,ht,6,t.R.g,16,1),i=0;i<u;i++)for(h=0;h<e.$o(t.R,D[i]);h++)r=e._o(t.R,D[i],h),(e.gq(t.R,r)||e.fm(t.R,r)==2||e.Vp(t.R,r))&&(P=e.Zo(t.R,D[i],h),n[r]||(o[Q++]=r,n[r]=true),s[P]==0&&(D[u++]=P,s[P]=L));y.sj(t.A,new e.kh(D,u,o,Q));}}},e.Cf=function(t){var A;for(t.S=j(l.Feb,ht,6,t.R.f,16,1),A=0;A<t.R.f;A++)if(e.Tl(t.R,A)==7){if(e.$o(t.R,A)==4){t.S[A]=true;continue;}if(e.$o(t.R,A)==3){if(e.Uo(t.R,A)==3){t.S[A]=true;continue;}if(e.Bl(t.R,A)==1){t.S[A]=true;continue;}if(e.$p(t.R,A,true))continue;if((t.Q&32)!=0){t.S[A]=true;continue;}if(e.eq(t.R,A)){t.S[A]=true;continue;}}}},e.Df=function(t,A){var n,P,r,i;for(r=false,P=0;P<t.R.g;P++)e.sf(t,P,1)&&(t.q[P]=t.L,A&&e.Sf(t,P),r=true);for(i=false,n=0;n<t.R.f;n++)e.xf(t,n,1)&&(t.ib[n]=t.L,A&&e.Tf(t,n),i=true);return i&&(t.L=!t.L),r||i;},e.Ef=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z;for(M=j(l.Feb,ht,6,t.R.f,16,1),p=j(l.Feb,ht,6,t.R.g,16,1),A=0,S=false,(t.Q&128)!=0&&(t.Y=j(l.j7,H,6,t.R.f,15,1),t.X=j(l.j7,H,6,t.R.g,15,1)),P=0;P<t.R.f;P++)t.W[P]&&(t.fb[P]||e.xf(t,P,1)&&(t.fb[P]=true,M[P]=true,++A));for(i=0;i<t.R.g;i++)t.V[i]&&(t.p[i]||e.sf(t,i,1)&&(t.p[i]=true,p[i]=true,++A));if(A==1){for(n=0;n<t.R.f;n++)if(M[n]){t.db[n]=0;break;}for(r=0;r<t.R.g;r++)if(p[r]){t.o[r]=0;break;}}else if(A>1)for(e.Bf(t),t.T=0,u=new y.Bqb(t.A);u.a<u.c.a.length;){for(D=y.Aqb(u),q=0,V=0,L=0,Q=0,s=-1,o=-1,c=0;c<D.a.length;c++)M[D.a[c]]&&(++q,(t.db[D.a[c]]==1||t.db[D.a[c]]==2)&&(++V,S=true,s<t.f[D.a[c]]&&(s=t.f[D.a[c]],L=D.a[c])));for(g=0;g<D.b.length;g++)p[D.b[g]]&&(++q,I=t.f[e.Zl(t.R,0,D.b[g])],Z=t.f[e.Zl(t.R,1,D.b[g])],h=I>Z?(I<<16)+Z:(Z<<16)+I,(t.o[D.b[g]]==1||t.o[D.b[g]]==2)&&(++V,S=true,o<h&&(o=h,Q=D.b[g])));if(q!=0)if(q==1){for(B=0;B<D.a.length;B++)M[D.a[B]]&&(t.db[D.a[B]]=0);for(b=0;b<D.b.length;b++)p[D.b[b]]&&(t.o[D.b[b]]=0);}else if(V==1){for(B=0;B<D.a.length;B++)M[D.a[B]]&&(t.db[D.a[B]]=3);for(b=0;b<D.b.length;b++)p[D.b[b]]&&(t.o[D.b[b]]=3);}else {if((t.Q&128)!=0){for(++t.T,B=0;B<D.a.length;B++)M[D.a[B]]&&(t.Y[D.a[B]]=t.T);for(b=0;b<D.b.length;b++)p[D.b[b]]&&(t.X[D.b[b]]=t.T);}if(k=false,s!=-1?t.db[L]==2&&(k=true):t.o[Q]==2&&(k=true),k){for(B=0;B<D.a.length;B++)if(M[D.a[B]])switch(t.db[D.a[B]]){case 1:t.db[D.a[B]]=2;break;case 2:t.db[D.a[B]]=1;}for(b=0;b<D.b.length;b++)if(p[D.b[b]])switch(t.o[D.b[b]]){case 1:t.o[D.b[b]]=2;break;case 2:t.o[D.b[b]]=1;}}}}return S;},e.Ff=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;for(i=y1(l.j7,[E1,H],[7,6],15,[2,32],2),D=0;D<2;D++){for(n=j(l.j7,E1,7,32,0,2),h=0,r=0;r<32;r++)if(A[D][r]!=null){for(L=A[D][r].length,n[r]=j(l.j7,H,6,L,15,1),u=0;u<L;u++)n[r][u]=t.f[A[D][r][u]];w.vzb(n[r],w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),++h;}for(s=h;s>0;s--){for(Q=0,o=null,P=0;P<32;P++)if(n[P]!=null){if(o==null||o.length<n[P].length)o=n[P],Q=P;else if(o.length==n[P].length){for(u=o.length-1;u>=0;u--)if(o[u]<n[P][u]){o=n[P],Q=P;break;}}}i[D][Q]=s,n[Q]=null;}}return i;},e.Gf=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p;if(s=false,t.R.K){for(Q=0;Q<t.R.g;Q++)if(e.hm(t.R,Q)!=0){s=true;break;}}for(t.O=2,n=0;n<t.R.f;n++)t.O=d.Math.max(t.O,e.$o(t.R,n)+e.rp(t.R,n));for(o=d.Math.max(2,s?(62+t.c+t.O*(t.c+23))/63|0:(62+t.c+t.O*(t.c+5))/63|0),t.f=j(l.j7,H,6,t.R.q,15,1),t.e=j(l.v7,He,101,t.R.f,0,1),P=0;P<t.R.f;P++)t.e[P]=new e.fh(o);for(u=false,r=0;r<t.R.f;r++)e.eh(t.e[r],r),pt(J(e.Ol(t.R,r),1),0)||e.Jl(t.R,r)!=null?e.ah(t.e[r],8,6):e.ah(t.e[r],8,Mt(e.Tl(t.R,r))),e.ah(t.e[r],8,Mt(e.Ml(t.R,r))),e.ah(t.e[r],2,Mt(e.Qo(t.R,r))),e.ah(t.e[r],4,Mt(e.$o(t.R,r)+e.rp(t.R,r))),pt(J(e.Ol(t.R,r),1),0)?e.ah(t.e[r],4,8):e.ah(t.e[r],4,Mt(8+e.Bl(t.R,r))),e.ah(t.e[r],5,Mt(d.Math.min(31,e.Uo(t.R,r)))),e.ah(t.e[r],4,Mt(e.vf(t,r)+1)),e.ah(t.e[r],2,Mt(e.Pl(t.R,r)>>4)),t.R.K&&(e.ah(t.e[r],51,e.Ol(t.R,r)),e.Jl(t.R,r)!=null&&(u=true));if(t.U=e.Kf(t),t.U<t.R.f){for(A=0;A<t.R.f;A++){for(e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),h=j(l.j7,H,6,e.$o(t.R,A),15,1),c=0;c<e.$o(t.R,A);c++)h[c]=t.f[e.Zo(t.R,A,c)]<<5,h[c]|=d.Math.min(31,e.Yo(t.R,e._o(t.R,A,c)));for(w.vzb(h,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),g=t.O;g>h.length;g--)e.ah(t.e[A],t.c+5,0);for(b=h.length-1;b>=0;b--)e.ah(t.e[A],t.c+5,l.Ueb(h[b]));}t.U=e.Kf(t);}if(u&&t.U<t.R.f){for(A=0;A<t.R.f;A++){for(e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),D=e.Jl(t.R,A),M=D==null?0:d.Math.min(12,D.length),c=12;c>M;c--)e.ah(t.e[A],8,0);for(b=M-1;b>=0;b--)e.ah(t.e[A],8,l.Ueb(D[b]));}t.U=e.Kf(t);}if(s&&t.U<t.R.f){for(A=0;A<t.R.f;A++){for(e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),L=j(l.k7,e0,6,e.$o(t.R,A)+e.rp(t.R,A),14,1),B=0,c=0;c<e.Mo(t.R,A);c++)(c<e.$o(t.R,A)||c>=e.Lo(t.R,A))&&(L[B]=Mt(t.f[e.Zo(t.R,A,c)]),L[B]=h1(L[B],23),L[B]=rt(L[B],Mt(e.hm(t.R,e._o(t.R,A,c)))),++B);for(w.vzb(L,w.aAb(h0(y.irb.prototype.ld,y.irb,[]))),g=t.O;g>L.length;g--)e.ah(t.e[A],t.c+23,0);for(b=L.length-1;b>=0;b--)e.ah(t.e[A],t.c+23,L[b]);}t.U=e.Kf(t);}if((t.Q&8)!=0&&t.U<t.R.f){for(k=new e.Zv(),i=0;i<t.R.f;i++)e.El(t.R,i)!=null&&e.Xv(k,e.El(t.R,i));for(A=0;A<t.R.f;A++)p=e.El(t.R,A)==null?0:1+e.Yv(k,e.El(t.R,A)),e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),e.ah(t.e[A],t.c,l.Ueb(p));t.U=e.Kf(t);}if((t.Q&16)!=0&&t.U<t.R.f){for(A=0;A<t.R.f;A++)e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),e.ah(t.e[A],1,Mt(e.Pm(t.R,A)?1:0));t.U=e.Kf(t);}(t.Q&512)!=0&&t.R.K&&e.jf(t);},e.Hf=function(t){var A,n,P,r,i,D,u,o,Q;for(n=0;n<t.R.f;n++)e.eh(t.e[n],n),e.ah(t.e[n],2*t.c+4,h1(Mt(t.f[n]),t.c+4));for(o=1;o<=t.U;o++){for(r=false,P=0;P<t.R.f;P++)t.f[P]==o&&(r=r|e.xf(t,P,2));if(r){if(u=t.U,t.U=e.Kf(t),t.U!=u)return  true;for(A=0;A<t.R.f;A++)e.eh(t.e[A],A),e.ah(t.e[A],2*t.c+4,h1(Mt(t.f[A]),t.c+4));}}for(Q=j(l.w7,He,173,t.R.g,0,1),D=0;D<Q.length;D++)Q[D]=new e.ih(t.f[e.Zl(t.R,0,D)],t.f[e.Zl(t.R,1,D)],D);for(y._qb(Q,0,Q.length,null),i=0;i<Q.length;i++)if(e.sf(t,Q[i].a,2)){for(;i+1<Q.length&&e.hh(Q[i],Q[i+1])==0;)e.sf(t,Q[++i].a,2);if(u=t.U,t.U=e.Kf(t),t.U!=u)return  true;for(A=0;A<t.R.f;A++)e.eh(t.e[A],A),e.ah(t.e[A],2*t.c+4,h1(Mt(t.f[A]),t.c+4));}return  false;},e.If=function(t){var A,n,P,r,i,D,u,o,Q,L;for(i=0,n=0;n<t.R.f;n++)t.bb[n]!=0&&(t.bb[n]!=2||(t.Q&256)==0)&&++i;if(i!=0){for(L=j(l.j7,H,6,i,15,1),i=0,P=0;P<t.R.f;P++)t.bb[P]!=0&&(t.bb[P]!=2||(t.Q&256)==0)&&(L[i]=t.bb[P]<<29|t.ab[P]<<24|t.f[P]<<12|P,++i);for(w.vzb(L,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),D=0,Q=0,u=L[0]&O0;;)if(++Q,Q==L.length||u!=(L[Q]&O0)){for(r=j(l.j7,H,6,Q-D,15,1),o=D;o<Q;o++)A=L[o]&4095,r[o-D]=A,t.gb[A]=true;if(y.sj(t.hb,r),Q==L.length)break;u=L[Q]&O0,D=Q;}}},e.Jf=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b;for(u=false,o=0;o<t.hb.a.length;o++){for(D=y.xj(t.hb,o),A=true,b=-1,Q=false,s=0;s<D.length;s++){if(n=D[s],t.db[n]==0){A=false;break;}if(t.db[n]!=3){for(L=true,h=0;h<D.length;h++)if(h!=s&&t.f[n]==t.f[D[h]]){L=false;break;}L&&b<t.f[n]&&(b=t.f[n],Q=t.db[n]==1);}}if(A&&b!=-1){for(P=D,r=0,i=P.length;r<i;++r)n=P[r],(t.db[n]==1||t.db[n]==2)&&(t.eb[n]=Q),t.gb[n]=false;y.Fj(t.hb,D),u=true,--o;}}return u;},e.Kf=function(t){var A,n;A=e.Af(t);do n=A,e.wf(t),A=e.Af(t);while(n!=A);return A;},e.Lf=function(t){for((t.Q&1)!=0&&(t.Q&2)==0&&(t.g=y.Jqb(t.f,t.R.f)),(t.Q&j1)==0&&(t.W=j(l.Feb,ht,6,t.R.f,16,1),t.V=j(l.Feb,ht,6,t.R.g,16,1),t.U<t.R.f&&(e.kf(t),(t.Q&j1)==0&&(e.Jf(t),t.P&&e.uh(t.P,t.f)))),t.g==null&&(t.Q&1)!=0&&(t.Q&2)!=0&&(t.g=y.Jqb(t.f,t.R.f));t.U<t.R.f;)e.lf(t),(t.Q&j1)==0&&(e.Jf(t),t.P&&e.uh(t.P,t.f));(t.Q&j1)==0&&(e.Jf(t),e.Ef(t),e.hg(t));},e.Mf=function(t){var A,n,P,r,i;for(i=t.U,r=y.Jqb(t.f,t.R.f),t.R.K||(e.Of(t),e.Cg(t,i,r)),t.bb=j(l.f7,Y,6,t.R.f,15,1),t.ab=j(l.f7,Y,6,t.R.f,15,1),n=0;n<t.R.f;n++)t.bb[n]=e.Hl(t.R,n)<<24>>24,t.ab[n]=e.Gl(t.R,n)<<24>>24;for(t.n=j(l.f7,Y,6,t.R.g,15,1),t.k=j(l.f7,Y,6,t.R.g,15,1),P=0;P<t.R.g;P++)t.n[P]=e.cm(t.R,P)<<24>>24,t.k[P]=e.bm(t.R,P)<<24>>24;for(e.Nf(t),t.Z=false,t.M=j(l.Feb,ht,6,t.R.f,16,1),A=0;A<t.R.f;A++)t.db[A]!=0&&(t.M[A]=true,t.Z=true);e.Qf(t),t.P=null,t.cb=j(l.Feb,ht,6,t.R.f,16,1),t.Z&&(t.P=new e.Ah(t.R,r,t.M,t.db,t.o,t.bb,t.ab,t.ib,t.q,t.cb),e.vh(t.P)),t.eb=j(l.Feb,ht,6,t.R.f,16,1),t.gb=j(l.Feb,ht,6,t.R.f,16,1),t.hb=new y.Mj(),t.P&&e.uh(t.P,t.f),e.If(t),e.Cg(t,i,r),e.Pf(t),t.P&&(t.K=e.rh(t.P)),e.$f(t);},e.Nf=function(t){var A,n,P,r,i,D;for(t.L=true,i=e.Df(t,false);t.U<t.R.f&&i;){for(A=0;A<t.R.f;A++)e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),D=t.db[A]<<7,(t.db[A]==1||t.db[A]==2)&&t.bb[A]!=0&&(D|=t.bb[A]<<5,D|=t.ab[A]),e.ah(t.e[A],18,h1(l.Ueb(D),9));for(n=0;n<t.R.g;n++)P=t.o[n]<<7,(t.o[n]==1||t.o[n]==2)&&e.im(t.R,n)==1&&t.n[n]!=0&&(P|=t.n[n]<<5,P|=t.k[n]),e.bh(t.e[e.Zl(t.R,0,n)],l.Ueb(P)),e.bh(t.e[e.Zl(t.R,1,n)],l.Ueb(P));if(r=e.Kf(t),t.U==r)break;t.U=r,i=e.Df(t,false);}},e.Of=function(t){var A,n,P,r;for(t.L=true,t.$=j(l.f7,Y,6,t.R.f,15,1),t.i=j(l.f7,Y,6,t.R.g,15,1),r=e.Df(t,true);t.U<t.R.f&&r;){for(A=0;A<t.R.f;A++)e.eh(t.e[A],A),e.ah(t.e[A],t.c+4,rt(h1(Mt(t.f[A]),4),Mt(t.db[A]<<2)));for(n=0;n<t.R.g;n++)e.bh(t.e[e.Zl(t.R,0,n)],Mt(t.o[n])),e.bh(t.e[e.Zl(t.R,1,n)],Mt(t.o[n]));if(P=e.Kf(t),t.U==P)break;t.U=P,r=e.Df(t,true);}},e.Pf=function(t){var A,n,P,r,i,D,u;for(t.L=true,P=e.Zf(t),t.P&&e.uh(t.P,t.f)&&(P=e.Zf(t)),e.Df(t,false)&&e.Jf(t),D=true;t.U<t.R.f&&D;){for(r=e.Ff(t,P),A=0;A<t.R.f;A++)e.eh(t.e[A],A),e.ah(t.e[A],t.c,Mt(t.f[A])),e.ah(t.e[A],20,0),!t.cb[A]&&t.bb[A]!=0&&e.bh(t.e[A],m1(Mt(t.bb[A]<<18),h1(Mt(r[t.bb[A]==1?0:1][t.ab[A]]),8))),u=t.db[A],t.eb[A]&&(u==1?u=2:u==2&&(u=1)),e.bh(t.e[A],l.Ueb(u<<4));for(n=0;n<t.R.g;n++)e.bh(t.e[e.Zl(t.R,0,n)],Mt(t.o[n])),e.bh(t.e[e.Zl(t.R,1,n)],Mt(t.o[n]));if(i=e.Kf(t),t.U==i)break;t.U=i,D=false,t.P&&e.uh(t.P,t.f)&&(D=true,P=e.Zf(t)),e.Df(t,false)&&(D=true,e.Jf(t));}},e.Qf=function(t){var A,n;for(A=0;A<t.R.f;A++)(!t.M[A]||t.db[A]==3)&&(t.bb[A]=0);for(n=0;n<t.R.g;n++)(e.im(t.R,n)!=1||t.o[n]==0||t.o[n]==3)&&(t.n[n]=0);},e.Rf=function(t,A,n){t.a==null&&(t.a=j(l.f7,Y,6,t.R.f,15,1),y.Mqb(t.a)),t.a[A]=n<<24>>24;},e.Sf=function(t,A){var n,P,r,i,D,u;if((t.o[A]==1||t.o[A]==2)&&!e.jq(t.R,A)){D=false;try{for(i=0;i<2;i++)if(n=e.Zl(t.R,i,A),e.$o(t.R,n)==3){for(P=j(l.j7,H,6,2,15,1),r=0,u=0;u<e.$o(t.R,n);u++)e._o(t.R,n,u)!=A&&(P[r++]=e.Zo(t.R,n,u));t.f[P[0]]>t.f[P[1]]^e.Uf(t,n,P[0],P[1])&&(D=!D);}}catch(o){if(o=M1(o),Kt(o,19)){t.i[A]=3;return;}else throw Ut(o);}t.o[A]==1^D?t.i[A]=1:t.i[A]=2;}},e.Tf=function(t,A){var n,P,r,i,D,u,o;if(t.db[A]==1||t.db[A]==2){if(u=false,e.Qo(t.R,A)==2&&e.$o(t.R,A)==2)try{for(D=0;D<2;D++)if(n=e.Zo(t.R,A,D),e.$o(t.R,n)==3){for(r=j(l.j7,H,6,2,15,1),i=0,o=0;o<e.$o(t.R,n);o++)e.ap(t.R,n,o)==1&&(r[i++]=e.Zo(t.R,n,o));t.f[r[0]]>t.f[r[1]]^e.Uf(t,n,r[0],r[1])&&(u=!u);}}catch(Q){if(Q=M1(Q),Kt(Q,19)){t.$[A]=3;return;}else throw Ut(Q);}else {try{P=e.Wf(t,A);}catch(Q){if(Q=M1(Q),Kt(Q,19)){t.$[A]=3;return;}else throw Ut(Q);}for(D=1;D<P.length;D++)for(o=0;o<D;o++)t.f[P[D]]<t.f[P[o]]&&(u=!u);}t.db[A]==1^u?t.$[A]=1:t.$[A]=2;}},e.Uf=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt;if(e.Tl(t.R,n)!=e.Tl(t.R,P))return e.Tl(t.R,n)>e.Tl(t.R,P);if(e.Ml(t.R,n)!=e.Ml(t.R,P))return it=e.Mm(t.R,n)?(e.Rk(),e.Pk)[e.Tl(t.R,n)]:e.Ml(t.R,n),Qt=e.Mm(t.R,P)?(e.Rk(),e.Pk)[e.Tl(t.R,P)]:e.Ml(t.R,P),it>Qt;for(V=t.R.f,M=j(l.j7,H,6,V,15,1),q=j(l.j7,H,6,V,15,1),S=j(l.j7,H,6,V,15,1),p=j(l.Feb,ht,6,V,16,1),o=j(l.Feb,ht,6,t.R.q,16,1),M[0]=A,M[1]=n,M[2]=P,q[0]=-1,q[1]=0,q[2]=0,o[A]=true,o[n]=true,o[P]=true,h=1,I=2,G=j(l.j7,H,6,64,15,1),G[1]=1,G[2]=3,c=2;h<=I;){for(;h<G[c];){if(b=M[h],!p[h]){for(g=0,B=0,z=0;z<e.$o(t.R,b);z++){if(L=e.Zo(t.R,b,z),I+e.ap(t.R,b,z)+1>=V&&(V+=t.R.f,M=e.Gg(M,V),q=e.Gg(q,V),S=e.Gg(S,V),p=e.Hg(p,V)),e.Yp(t.R,e._o(t.R,b,z)))++g,B+=e.Tl(t.R,L);else if(L!=A)for(ot=1;ot<e.ap(t.R,b,z);ot++)++I,M[I]=L,q[I]=h,p[I]=true;if(Dt=q[h],L!=M[Dt]){if(u=false,o[L])for(ut=q[Dt];ut!=-1;){if(L==M[ut]){u=true;break;}ut=q[ut];}u?(++I,M[I]=L,q[I]=h,p[I]=true):(++I,M[I]=L,q[I]=h,o[L]=true);}}g!=0&&(++I,S[I]=(B<<2)/g|0,q[I]=h,p[I]=true);}if(++h,h==K0)throw Ut(new T.cC("Emergency break in while loop."));}for(G.length==c+1&&(G=e.Gg(G,G.length+64)),G[c+1]=I+1,Z=G[c];Z<G[c+1];Z++)S[Z]==0&&(S[Z]=(e.Tl(t.R,M[Z])==151||e.Tl(t.R,M[Z])==152?1:e.Tl(t.R,M[Z]))<<2),S[Z]+=S[q[Z]]<<16;if(e.Yf(t,p,S,q,M,G,c),S[1]!=S[2])return S[1]>S[2];c>1&&e.Vf(S,q,G,c),++c;}for(s=j(l.j7,H,6,t.R.f,15,1),tt=false,i=0;i<t.R.f;i++)if(o[i]&&!e.Mm(t.R,i)){tt=true;break;}if(tt){for(D=0;D<t.R.f;D++)s[D]=e.Mm(t.R,D)?(e.Rk(),e.Pk)[e.Tl(t.R,D)]:e.Ml(t.R,D);if(e.Xf(t,p,S,q,M,s,G,c))return S[1]>S[2];}for(y.Tqb(s,s.length,0),k=false,Q=0;Q<t.R.g;Q++)(o[e.Zl(t.R,0,Q)]||o[e.Zl(t.R,1,Q)])&&(t.i[Q]==1?(s[e.Zl(t.R,0,Q)]=1,s[e.Zl(t.R,1,Q)]=1,k=true):t.i[Q]==2&&(s[e.Zl(t.R,0,Q)]=2,s[e.Zl(t.R,1,Q)]=2,k=true));if(k&&e.Xf(t,p,S,q,M,s,G,c))return S[1]>S[2];for(y.Tqb(s,s.length,0),lt=false,r=0;r<t.R.f;r++)o[r]&&(t.$[r]==2?(s[r]=1,lt=true):t.$[r]==1&&(s[r]=2,lt=true));if(lt&&e.Xf(t,p,S,q,M,s,G,c))return S[1]>S[2];throw t.d=true,Ut(new T.cC("no distinction applying CIP rules"));},e.Vf=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h;for(s=n[P],D=n[P+1]-s,h=j(l.t7,C1,175,D,0,1),o=0;o<D;o++)h[o]=new e._g(),h[o].c=t[o+s],h[o].b=A[o+s],h[o].a=o+s;for(r=new e.Yg(),L=P;L>1;L--){for(Q=0;Q<D;Q++)h[Q].c+=t[h[Q].b]<<16,h[Q].b=A[h[Q].b];for(y._qb(h,0,h.length,r),i=1,u=0;u<D;u++)t[h[u].a]=i,u!=D-1&&e.Xg(h[u],h[u+1])!=0&&++i;}},e.Wf=function(t,A){var n,P,r,i,D,u,o;for(D=e.Lo(t.R,A),u=j(l.j7,H,6,D,15,1),r=0;r<D;r++)u[r]=e.Zo(t.R,A,r);for(P=D;P>1;P--){for(n=false,i=1;i<P;i++)e.Uf(t,A,u[i-1],u[i])&&(n=true,o=u[i-1],u[i-1]=u[i],u[i]=o);if(!n)break;}return u;},e.Xf=function(t,A,n,P,r,i,D,u){var o,Q;for(Q=1;Q<u;Q++){for(o=D[Q];o<D[Q+1];o++)n[o]=i[r[o]]+(n[P[o]]<<8);if(e.Yf(t,A,n,P,r,D,Q),n[1]!=n[2])return  true;Q>1&&e.Vf(n,P,D,Q);}return  false;},e.Yf=function(t,A,n,P,r,i,D){var u,o,Q,L,s,h,b,c,g,B,k;for(s=D;s>1;s--){for(g=i[s]-i[s-1],k=j(l.r7,C1,174,g,0,1),u=i[s],c=0;c<g;c++){for(B=i[s-1]+c,h=u;h<i[s+1]&&P[h]==B;)++h;for(k[c]=new e.Vg(),k[c].c=B,k[c].d=n[B],k[c].b=A[B]?0:e.Cp(t.R,r[B]),k[c].a=j(l.j7,H,6,h-u,15,1),L=u;L<h;L++)k[c].a[L-u]=n[L];w.vzb(k[c].a,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),u=h;}for(o=new e.Sg(),y._qb(k,0,k.length,o),Q=1,b=0;b<g;b++)n[k[b].c]=Q,b!=g-1&&e.Rg(k[b],k[b+1])!=0&&++Q;}},e.Zf=function(t){var A,n;for(n=y1(l.j7,[c0,E1],[28,7],0,[2,32],2),A=0;A<t.R.f;A++)t.M[A]&&(t.bb[A]==1?n[0][t.ab[A]]=e.Bh(n[0][t.ab[A]],A):t.bb[A]==2&&(n[1][t.ab[A]]=e.Bh(n[0][t.ab[A]],A)));return n;},e.$f=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h;for(i=0,L=0,D=0,u=0,o=0,Q=0,s=0,h=false,A=j(l.Feb,ht,6,32,16,1),n=0;n<t.R.f;n++)t.db[n]!=0&&(++i,t.db[n]==3?++L:t.bb[n]==0?(++D,t.P&&e.qh(t.P,n)&&++u):t.bb[n]==2?t.ab[n]==0&&++Q:t.bb[n]==1&&(r=t.ab[n],A[r]||(++s,A[r]=true),t.ab[n]==0&&++o,t.P&&e.qh(t.P,n)&&(h=true)));for(P=0;P<t.R.g;P++)t.o[P]!=0&&e.im(t.R,P)==1&&(++i,t.o[P]==3?++L:t.n[P]==0?(++D,t.P&&e.qh(t.P,e.Zl(t.R,0,P))&&e.qh(t.P,e.Zl(t.R,1,P))&&++u):t.n[P]==2?t.k[P]==0&&++Q:t.n[P]==1&&(r=t.k[P],A[r]||(++s,A[r]=true),t.k[P]==0&&++o,t.P&&e.qh(t.P,e.Zl(t.R,0,P))&&e.qh(t.P,e.Zl(t.R,1,P))&&(h=true)));if(i==0){e.Nn(t.R,q1);return;}if(L!=0){e.Nn(t.R,0);return;}if(t.K){e.Nn(t.R,n0+(1<<s));return;}o+u==i&&!h?e.Nn(t.R,Yn):D==i?e.Nn(t.R,sA):Q==i?e.Nn(t.R,327680):D==i-1&&o==1?e.Nn(t.R,te):e.Nn(t.R,458752+(1<<s));},e._f=function(t,A,n){for(;n!=0;)t.t==0&&((!t.r||t.v!=63)&&(t.v+=64),T.Ojb(t.u,t.v&n1),t.t=6,t.v=0),t.v<<=1,t.v=Jt(rt(Mt(t.v),J(A,1))),A=t0(A,1),--n,--t.t;},e.ag=function(t){return t.v<<=t.t,(!t.r||t.v!=63)&&(t.v+=64),T.Ojb(t.u,t.v&n1),t.u.a;},e.bg=function(t,A){t.u=new T.Ujb(),t.t=6,t.v=0,t.r=A;},e.cg=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B;if(t.R.f==0){t.s="";return;}for(s=t.jb&t.b,B=t.jb?16:8,e.bg(t,true),T.Ojb(t.u,s?35:33),e._f(t,Mt(t.jb?1:0),1),e._f(t,l.Ueb(A?1:0),1),e._f(t,l.Ueb(B>>1),4),c=0,o=1;o<t.R.f;o++)c=e.sg(t,t.B[o],t.F[o]==-1?-1:t.B[t.F[o]],c,n);if(s)for(Q=0;Q<t.R.f;Q++)for(P=t.B[Q],b=e.$o(t.R,P);b<e.Lo(t.R,P);b++)c=e.sg(t,e.Zo(t.R,P,b),P,c,n);if(t.R.f>1&&c==0){t.s="";return;}for(D=1<<B,h=c/((D>>1)-1),g=c+.5*h,L=1;L<t.R.f;L++)e.dg(t,t.B[L],t.F[L]==-1?-1:t.B[t.F[L]],g,h,B,n);if(s)for(u=0;u<t.R.f;u++)for(P=t.B[u],b=e.$o(t.R,P);b<e.Lo(t.R,P);b++)e.dg(t,e.Zo(t.R,P,b),P,g,h,B,n);A&&(i=t.jb?1.5:(e.Rk(),e.Rk(),e.Qk),r=e.Xl(t.R,s?t.R.q:t.R.f,s?t.R.r:t.R.g,i,n),e._f(t,Mt(d.Math.min(D-1,d.Math.max(0,Rt(.5+d.Math.log(r/.1)*d.Math.LOG10E/(d.Math.log(2e3)*d.Math.LOG10E)*(D-1))))),B),e._f(t,Mt(e.gg(n[t.B[0]].a/r,D)),B),e._f(t,Mt(e.gg(n[t.B[0]].b/r,D)),B),t.jb&&e._f(t,Mt(e.gg(n[t.B[0]].c/r,D)),B)),t.s=e.ag(t);},e.dg=function(t,A,n,P,r,i,D){var u,o,Q;u=n==-1?(D[A].a-D[t.B[0]].a)/8:D[A].a-D[n].a,o=n==-1?(D[A].b-D[t.B[0]].b)/8:D[A].b-D[n].b,e._f(t,Mt(Rt((P+u)/r)),i),e._f(t,Mt(Rt((P+o)/r)),i),t.jb&&(Q=n==-1?(D[A].c-D[t.B[0]].c)/8:D[A].c-D[n].c,e._f(t,Mt(Rt((P+Q)/r)),i));},e.eg=function(t,A){var n;for(n=0;n<t.w;n++)A-=16;for(A<0&&T.Zjb();A>15;)e._f(t,1,1),e._f(t,15,4),A-=16,++t.w;e._f(t,1,1),e._f(t,l.Ueb(A),4);},e.fg=function(t){var A,n,P,r,i,D;if(t.R.f==0){t.N="";return;}for(i=0,P=false,r=false,n=0;n<t.R.f;n++)i<e.Ll(t.R,n)&&(i=e.Ll(t.R,n)),e.Am(t.R,n)?P=true:r=true;if(i==0){t.N="";return;}for(D=e.Pg(i),e.bg(t,true),e._f(t,l.Ueb(D),4),e._f(t,l.Ueb(P?1:0),1),e._f(t,l.Ueb(r?1:0),1),A=0;A<t.R.f;A++)e._f(t,Mt(e.Ll(t.R,t.B[A])),D),P&&r&&e._f(t,Mt(e.Am(t.R,t.B[A])?1:0),1);t.N=e.ag(t);},e.gg=function(t,A){var n,P,r,i;return n=A/2|0,r=t<0,t=d.Math.abs(t),i=A/32|0,P=d.Math.min(n-1,Jt(c1(d.Math.round(t*n/(t+i))))),r?n+P:P;},e.hg=function(t){var A,n,P;for(A=0;A<t.R.f;A++)t.db[A]==3&&!e.wm(t.R,A)&&e.Vn(t.R,A),(e.Hl(t.R,A)==1||e.Hl(t.R,A)==2)&&t.db[A]==3&&e.Vn(t.R,A),e.wm(t.R,A)&&t.db[A]!=3&&!e.Eg(t,A)&&e.Vn(t.R,A);for(P=0;P<t.R.r;P++)e.Rm(t.R,P)&&!e.Dg(t,P)&&e.Vn(t.R,e.Zl(t.R,0,P));for(n=0;n<t.R.g;n++)e.fm(t.R,n)==2&&(e.Gm(t.R,n)&&(t.o[n]==1||t.o[n]==2)&&(t.o[n]=3,e.Mn(t.R,n,386)),t.o[n]==3&&!t.p[n]&&e.im(t.R,n)!=386&&(e.Vn(t.R,e.Zl(t.R,0,n)),e.Vn(t.R,e.Zl(t.R,1,n)))),e.im(t.R,n)==1&&t.o[n]==3&&!e.wm(t.R,e.Zl(t.R,0,n))&&!e.wm(t.R,e.Zl(t.R,1,n))&&(e.Vn(t.R,e.Zl(t.R,0,n)),e.Vn(t.R,e.Zl(t.R,1,n))),(e.cm(t.R,n)==1||e.cm(t.R,n)==2)&&(e.im(t.R,n)!=1||t.o[n]!=1&&t.o[n]!=2)&&(e.Vn(t.R,e.Zl(t.R,0,n)),e.Vn(t.R,e.Zl(t.R,1,n)));},e.ig=function(t){var A,n,P,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;if(t.R.f!=0&&!t.G){for(t.I=0,S=0,n=1;n<t.R.f;n++)t.f[n]>t.f[S]&&(S=n);for(P=j(l.Feb,ht,6,t.R.f,16,1),D=j(l.Feb,ht,6,t.R.g,16,1),t.H=j(l.j7,H,6,t.R.f,15,1),t.B=j(l.j7,H,6,t.R.f,15,1),t.F=j(l.j7,H,6,t.R.f,15,1),t.C=j(l.j7,H,6,t.R.g,15,1),t.B[0]=S,t.H[S]=0,P[S]=true,o=0,Q=1,L=0;o<t.R.f;)if(o<Q){for(;;){for(c=0,g=0,h=-1,A=t.B[o],B=0;B<e.Mo(t.R,A);B++)(B<e.$o(t.R,A)||B>=e.Lo(t.R,A))&&(u=e.Zo(t.R,A,B),!P[u]&&t.f[u]>h&&(c=u,g=e._o(t.R,A,B),h=t.f[u]));if(h==-1)break;t.H[c]=Q,t.F[Q]=o,t.B[Q++]=c,t.C[L++]=g,P[c]=true,D[g]=true;}++o;}else {for(b=0,h=-1,A=0;A<t.R.f;A++)!P[A]&&t.f[A]>h&&(b=A,h=t.f[A]);t.H[b]=Q,t.F[Q]=-1,t.B[Q++]=b,P[b]=true;}for(t.D=j(l.j7,H,6,2*(t.R.g-L),15,1);;){for(M=t.R.M,p=t.R.M,q=-1,i=0;i<t.R.g;i++)D[i]||(t.H[e.Zl(t.R,0,i)]<t.H[e.Zl(t.R,1,i)]?(k=t.H[e.Zl(t.R,0,i)],s=t.H[e.Zl(t.R,1,i)]):(k=t.H[e.Zl(t.R,1,i)],s=t.H[e.Zl(t.R,0,i)]),(k<M||k==M&&s<p)&&(M=k,p=s,q=i));if(q==-1)break;D[q]=true,t.C[L++]=q,t.D[2*t.I]=M,t.D[2*t.I+1]=p,++t.I;}t.G=true;}},e.jg=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c;for(o=null,c=e.Ep(t.R),s=0;s<c.j.a.length;s++)if(c.f[s]){for(i=0,h=y.xj(c.j,s),n=h,P=0,r=n.length;P<r;++P)A=n[P],e.yg(t,A)&&++i;if(i!=0)if(b=y.xj(c.k,s),o==null&&(o=j(l.Feb,ht,6,t.R.g,16,1)),i==h.length){for(Q=-1,L=St,D=0;D<h.length;D++)L>t.B[b[D]]&&(L=t.B[b[D]],Q=D);for(;i>0;)o[b[Q]]=true,Q=e.Mg(Q+2,h.length),i-=2;}else {for(u=0;e.yg(t,h[u]);)++u;for(;!e.yg(t,h[u]);)u=e.Mg(u+1,h.length);for(;i>0;)for(o[b[u]]=true,u=e.Mg(u+2,h.length),i-=2;!e.yg(t,h[u]);)u=e.Mg(u+1,h.length);}}return o;},e.kg=function(t,A){var n,P,r,i,D,u,o,Q,L;for(e.ig(t),Q=new e.uw(t.R.f,t.R.g),e.On(Q,t.R.K),D=0;D<t.R.f;D++)e.ll(t.R,Q,t.B[D],0,0),e.on(Q,D,t.bb[t.B[D]],t.ab[t.B[D]]);for(u=0;u<t.R.g;u++)e.nl(t.R,Q,t.C[u],0,0,t.H,false),!(Q.H[u]==257||Q.H[u]==129)&&Q.D[0][u]>Q.D[1][u]&&(L=Q.D[0][u],e.Dn(Q,0,u,Q.D[1][u]),Q.D[1][u]=L,Q.T=0),e.Gn(Q,u,t.n[t.C[u]],t.k[t.C[u]]);if(A)for(i=0;i<t.R.f;i++)for(n=t.B[i],o=e.$o(t.R,n);o<e.Lo(t.R,n);o++)r=e.ll(t.R,Q,e.Zo(t.R,n,o),0,0),e.ml(t.R,Q,e._o(t.R,n,o),0,0,t.H[n],r,false);for(P=0;P<Q.r;P++)n=Q.D[0][P],t.eb[t.B[n]]&&(Q.H[P]==257?(Q.H[P]=129,Q.T=0):Q.H[P]==129&&(Q.H[P]=257,Q.T=0));return e.aw(t.R,Q),Q.T&=-9,Q;},e.lg=function(t,A){return t.o[A];},e.mg=function(t){return e.ng(t,t.jb);},e.ng=function(t,A){return t.s==null&&(e.ig(t),e.cg(t,A,t.R.J)),t.s;},e.og=function(t){return t.N==null&&(e.ig(t),e.fg(t)),t.N;},e.pg=function(t){return e.ig(t),t.B;},e.qg=function(t){return e.ig(t),t.H;},e.rg=function(t){return t.J==null&&(e.ig(t),(t.Q&j1)==0&&(e.Ag(t),e.Bg(t,1),e.Bg(t,2)),e.zg(t)),t.J;},e.sg=function(t,A,n,P,r){var i,D,u;return i=n==-1?d.Math.abs(r[A].a-r[t.B[0]].a)/8:d.Math.abs(r[A].a-r[n].a),P<i&&(P=i),D=n==-1?d.Math.abs(r[A].b-r[t.B[0]].b)/8:d.Math.abs(r[A].b-r[n].b),P<D&&(P=D),t.jb&&(u=n==-1?d.Math.abs(r[A].c-r[t.B[0]].c)/8:d.Math.abs(r[A].c-r[n].c),P<u&&(P=u)),P;},e.tg=function(t,A){return t.X[A];},e.ug=function(t,A){return t.Y[A];},e.vg=function(t,A){return t.g==null?-1:t.g[A];},e.wg=function(t,A){return t.db[A];},e.xg=function(t,A){var n,P,r,i,D,u,o;for(o=e.Ep(t.R),n=0;n<o.j.a.length;n++)if(o.e[n]&&e.qt(o,n,A)){for(r=y.xj(o.j,n),i=0,D=r.length;i<D;++i)if(P=r[i],P!=A){for(u=0;u<e.$o(t.R,P);u++)if(e.Vp(t.R,e._o(t.R,P,u)))return  true;}return  false;}return  false;},e.yg=function(t,A){var n,P,r;if(e.Qo(t.R,A)<2)return  false;if(e.$o(t.R,A)==2)return  true;for(n=0,r=0;r<e.$o(t.R,A);r++)P=e._o(t.R,A,r),e.Up(t.R,P)&&(n+=e.fm(t.R,P)-1);return n>1;},e.zg=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt,b1,u1,Ht,gt;if(e.bg(t,false),e._f(t,9,4),jt=d.Math.max(e.Pg(t.R.f),e.Pg(t.R.g)),e._f(t,l.Ueb(jt),4),jt==0){e._f(t,Mt(t.R.K?1:0),1),e._f(t,0,1),t.J=e.ag(t);return;}for(b1=Ht=u1=it=0,o=0;o<t.R.f;o++)if(H1(J(e.Ol(t.R,o),1),0)){switch(e.Tl(t.R,o)){case 6:break;case 7:++b1;break;case 8:++Ht;break;default:++u1;}e.Bl(t.R,o)!=0&&++it;}for(e._f(t,Mt(t.R.f),jt),e._f(t,Mt(t.R.g),jt),e._f(t,l.Ueb(b1),jt),e._f(t,l.Ueb(Ht),jt),e._f(t,l.Ueb(u1),jt),e._f(t,l.Ueb(it),jt),Q=0;Q<t.R.f;Q++)e.Tl(t.R,t.B[Q])==7&&H1(J(e.Ol(t.R,t.B[Q]),1),0)&&e._f(t,l.Ueb(Q),jt);for(h=0;h<t.R.f;h++)e.Tl(t.R,t.B[h])==8&&H1(J(e.Ol(t.R,t.B[h]),1),0)&&e._f(t,l.Ueb(h),jt);for(b=0;b<t.R.f;b++)e.Tl(t.R,t.B[b])!=6&&e.Tl(t.R,t.B[b])!=7&&e.Tl(t.R,t.B[b])!=8&&H1(J(e.Ol(t.R,t.B[b]),1),0)&&(e._f(t,l.Ueb(b),jt),e._f(t,Mt(e.Tl(t.R,t.B[b])),8));for(c=0;c<t.R.f;c++)e.Bl(t.R,t.B[c])!=0&&H1(J(e.Ol(t.R,t.B[c]),1),0)&&(e._f(t,l.Ueb(c),jt),e._f(t,Mt(8+e.Bl(t.R,t.B[c])),4));for(Bt=0,S=0,g=1;g<t.R.f;g++)t.F[g]==-1?yt=0:(yt=1+t.F[g]-S,S=t.F[g]),Bt<yt&&(Bt=yt);for(ct=e.Pg(Bt),e._f(t,l.Ueb(ct),4),S=0,B=1;B<t.R.f;B++)t.F[B]==-1?yt=0:(yt=1+t.F[B]-S,S=t.F[B]),e._f(t,l.Ueb(yt),ct);for(xt=0;xt<2*t.I;xt++)e._f(t,Mt(t.D[xt]),jt);for(I=0;I<t.R.g;I++)G=(e.hm(t.R,t.C[I])&UA)!=0||e.im(t.R,t.C[I])==16?1:e.Yp(t.R,t.C[I])?0:d.Math.min(3,e.fm(t.R,t.C[I])),e._f(t,l.Ueb(G),2);if(n=0,(t.Q&j1)==0)for(k=0;k<t.R.f;k++)t._[t.B[k]]!=0&&t._[t.B[k]]!=3&&++n;if(e._f(t,l.Ueb(n),jt),(t.Q&j1)==0)for(k=0;k<t.R.f;k++)t._[t.B[k]]!=0&&t._[t.B[k]]!=3&&(e._f(t,l.Ueb(k),jt),t.bb[t.B[k]]==0?e._f(t,Mt(t._[t.B[k]]),3):(gt=t._[t.B[k]]==1?t.bb[t.B[k]]==1?4:6:t.bb[t.B[k]]==1?5:7,e._f(t,l.Ueb(gt),3),e._f(t,Mt(t.ab[t.B[k]]),3)));if(A=0,(t.Q&j1)==0)for(Z=0;Z<t.R.g;Z++)t.j[t.C[Z]]!=0&&t.j[t.C[Z]]!=3&&(!e.jq(t.R,t.C[Z])||e.im(t.R,t.C[Z])==1)&&++A;if(e._f(t,l.Ueb(A),jt),(t.Q&j1)==0)for(Z=0;Z<t.R.g;Z++)t.j[t.C[Z]]!=0&&t.j[t.C[Z]]!=3&&(!e.jq(t.R,t.C[Z])||e.im(t.R,t.C[Z])==1)&&(e._f(t,l.Ueb(Z),jt),e.im(t.R,t.C[Z])==1?t.n[t.C[Z]]==0?e._f(t,Mt(t.j[t.C[Z]]),3):(gt=t.j[t.C[Z]]==1?t.n[t.C[Z]]==1?4:6:t.n[t.C[Z]]==1?5:7,e._f(t,l.Ueb(gt),3),e._f(t,Mt(t.k[t.C[Z]]),3)):e._f(t,Mt(t.j[t.C[Z]]),2));for(e._f(t,Mt(t.R.K?1:0),1),Dt=0,M=0;M<t.R.f;M++)e.Ml(t.R,t.B[M])!=0&&++Dt;if(Dt!=0)for(e.eg(t,1),e._f(t,l.Ueb(Dt),jt),k=0;k<t.R.f;k++)e.Ml(t.R,t.B[k])!=0&&(e._f(t,l.Ueb(k),jt),e._f(t,Mt(e.Ml(t.R,t.B[k])),8));if(t.w=0,t.R.K){for(e.gf(t,0,jt,j1,1,-1),e.gf(t,3,jt,Z1,1,-1),e.gf(t,4,jt,120,4,3),e.gf(t,5,jt,Xn,2,1),e.gf(t,6,jt,1,1,-1),e.gf(t,7,jt,X0,4,7),Dt=0,k=0;k<t.R.f;k++)e.Jl(t.R,t.B[k])!=null&&++Dt;if(Dt>0){for(e.eg(t,8),e._f(t,l.Ueb(Dt),jt),p=0;p<t.R.f;p++)if(q=e.Jl(t.R,t.B[p]),q!=null)for(e._f(t,l.Ueb(p),jt),e._f(t,Mt(q.length),4),r=q,i=0,D=r.length;i<D;++i)P=r[i],e._f(t,l.Ueb(P),8);}e.hf(t,9,jt,384,2,7),e.hf(t,10,jt,31,5,0),e.gf(t,11,jt,L0,1,-1),e.hf(t,12,jt,UA,8,9),e.gf(t,13,jt,$A,3,14),e.gf(t,14,jt,aA,5,17),e.gf(t,16,jt,kA,3,22);}for(Dt=0,L=0;L<t.R.f;L++)t.a!=null&&t.a[t.B[L]]!=-1&&++Dt;if(Dt!=0)for(e.eg(t,17),e._f(t,l.Ueb(Dt),jt),k=0;k<t.R.f;k++)t.a!=null&&t.a[t.B[k]]!=-1&&(e._f(t,l.Ueb(k),jt),e._f(t,Mt(t.a[t.B[k]]),4));if((t.Q&8)!=0||(t.Q&i0)!=0){for(Dt=0,It=0,k=0;k<t.R.f;k++)Zt=e.El(t.R,t.B[k]),Zt!=null&&(++Dt,It=d.Math.max(It,w.aAb(Zt).length));if(Dt!=0){for(Nt=e.Pg(It),e.eg(t,18),e._f(t,l.Ueb(Dt),jt),e._f(t,l.Ueb(Nt),4),p=0;p<t.R.f;p++)if(lt=e.El(t.R,t.B[p]),lt!=null)for(e._f(t,l.Ueb(p),jt),e._f(t,Mt(w.aAb(lt).length),Nt),bt=0;bt<w.aAb(lt).length;bt++)e._f(t,(w.Xzb(bt,w.aAb(lt).length),Mt(w.aAb(lt).charCodeAt(bt))),7);}}for(t.R.K&&(e.gf(t,19,jt,mA,3,25),e.hf(t,20,jt,iA,3,17)),Dt=0,s=0;s<t.R.f;s++)e.Pl(t.R,t.B[s])!=0&&++Dt;if(Dt!=0)for(e.eg(t,21),e._f(t,l.Ueb(Dt),jt),u=0;u<t.R.f;u++)e.Pl(t.R,t.B[u])!=0&&(e._f(t,l.Ueb(u),jt),e._f(t,Mt(e.Pl(t.R,t.B[u])>>4),2));if(t.R.K&&(e.gf(t,22,jt,IA,1,-1),e.hf(t,23,jt,hA,1,-1),e.hf(t,24,jt,on,2,21)),(t.Q&16)!=0){for(u=0;u<t.R.f;u++)if(e.Pm(t.R,t.B[u])){for(e.eg(t,25),P=0;P<t.R.f;P++)e._f(t,Mt(e.Pm(t.R,t.B[P])?1:0),1);break;}}if(Et=e.jg(t),Et!=null){for(Dt=0,Z=0;Z<t.R.g;Z++)Et[t.C[Z]]&&++Dt;for(e.eg(t,26),e._f(t,l.Ueb(Dt),jt),z=0;z<t.R.g;z++)Et[t.C[z]]&&e._f(t,l.Ueb(z),jt);}for(t.R.K&&e.gf(t,27,jt,r0,1,-1),Dt=0,tt=0;tt<t.R.g;tt++)e.im(t.R,t.C[tt])==16&&++Dt;if(Dt!=0)for(e.eg(t,28),e._f(t,l.Ueb(Dt),jt),Z=0;Z<t.R.g;Z++)e.im(t.R,t.C[Z])==16&&e._f(t,l.Ueb(Z),jt);for(t.R.K&&(e.gf(t,29,jt,TA,2,30),e.gf(t,30,jt,EA,7,32),e.gf(t,32,jt,Dr,2,44),e.gf(t,33,jt,Qe,5,39),e.gf(t,34,jt,en,1,-1),e.hf(t,35,jt,Le,1,-1),e.hf(t,36,jt,96,2,5)),Dt=0,ot=0;ot<t.R.g;ot++)(e.im(t.R,t.C[ot])==32||e.im(t.R,t.C[ot])==64)&&++Dt;if(Dt!=0)for(e.eg(t,37),e._f(t,l.Ueb(Dt),jt),V=0;V<t.R.g;V++)(e.im(t.R,t.C[V])==32||e.im(t.R,t.C[V])==64)&&(e._f(t,l.Ueb(V),jt),e._f(t,Mt(e.im(t.R,t.C[V])==32?0:1),1));if(t.b&&(t.Q&16)!=0){for(Dt=0,Qt=0,k=0;k<t.R.f;k++){for(ut=0,qt=e.$o(t.R,t.B[k]);qt<e.Lo(t.R,t.B[k]);qt++)e.Pm(t.R,e.Zo(t.R,t.B[k],qt))&&(wt=qt-e.$o(t.R,t.B[k]),ut|=1<<wt,Qt=d.Math.max(Qt,wt+1));ut!=0&&++Dt;}if(Dt!=0)for(e.eg(t,38),e._f(t,l.Ueb(Dt),jt),e._f(t,l.Ueb(Qt),3),u=0;u<t.R.f;u++){for(ut=0,qt=e.$o(t.R,t.B[u]);qt<e.Lo(t.R,t.B[u]);qt++)e.Pm(t.R,e.Zo(t.R,t.B[u],qt))&&(wt=qt-e.$o(t.R,t.B[u]),ut|=1<<wt);ut!=0&&(e._f(t,l.Ueb(u),jt),e._f(t,l.Ueb(ut),Qt));}}t.R.K&&e.gf(t,39,jt,ur,4,47),e._f(t,0,1),t.J=e.ag(t);},e.Ag=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(t._=j(l.f7,Y,6,t.R.f,15,1),A=0;A<t.R.f;A++)if(t.db[A]==1||t.db[A]==2){if(o=t.eb[A],e.Wp(t.R,A))for(u=0;u<e.$o(t.R,A);u++){for(r=e.Zo(t.R,A,u),s=0,L=j(l.j7,H,6,3,15,1),Q=0;Q<e.$o(t.R,r);Q++)L[s]=e.Zo(t.R,r,Q),L[s]!=A&&++s;s==2&&t.f[L[0]]>t.f[L[1]]^t.H[L[0]]<t.H[L[1]]&&(o=!o);}else for(u=1;u<e.$o(t.R,A);u++)for(Q=0;Q<u;Q++)i=e.Zo(t.R,A,u),D=e.Zo(t.R,A,Q),t.f[i]>t.f[D]&&(o=!o),t.H[i]<t.H[D]&&(o=!o);t._[A]=t.db[A]==1^o?1:2;}else t._[A]=t.db[A];for(t.j=j(l.f7,Y,6,t.R.g,15,1),n=0;n<t.R.g;n++)if(t.o[n]==1||t.o[n]==2){for(o=false,u=0;u<2;u++)if(P=e.Zl(t.R,u,n),e.$o(t.R,P)==3){for(L=j(l.j7,H,6,2,15,1),s=0,Q=0;Q<3;Q++)e.Zo(t.R,P,Q)!=e.Zl(t.R,1-u,n)&&(L[s++]=e.Zo(t.R,P,Q));t.f[L[0]]>t.f[L[1]]&&(o=!o),t.H[L[0]]<t.H[L[1]]&&(o=!o);}t.j[n]=t.o[n]==1^o?1:2;}else t.j[n]=t.o[n];},e.Bg=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c;for(o=j(l.j7,H,6,32,15,1),Q=0,P=0;P<t.R.f;P++)(t._[P]==1||t._[P]==2)&&t.bb[P]==A&&(u=t.ab[P],o[u]<t.f[P]&&(o[u]==0&&++Q,o[u]=t.f[P]));for(i=0;i<t.R.g;i++)(t.j[i]==1||t.j[i]==2)&&t.n[i]==A&&e.im(t.R,i)==1&&(u=t.k[i],c=d.Math.max(t.f[e.Zl(t.R,0,i)],t.f[e.Zl(t.R,1,i)]),o[u]<c&&(o[u]==0&&++Q,o[u]=c));for(D=j(l.f7,Y,6,32,15,1),L=0;L<Q;L++){for(h=-1,b=0,s=0;s<32;s++)b<o[s]&&(b=o[s],h=s);o[h]=0,D[h]=L<<24>>24;}for(n=0;n<t.R.f;n++)(t._[n]==1||t._[n]==2)&&t.bb[n]==A&&(t.ab[n]=D[t.ab[n]]);for(r=0;r<t.R.g;r++)(t.j[r]==1||t.j[r]==2)&&t.n[r]==A&&e.im(t.R,r)==1&&(t.k[r]=D[t.k[r]]);},e.Cg=function(t,A,n){var P,r;for(t.U=A,P=0;P<t.R.f;P++)t.f[P]=n[P],t.db[P]=0,t.ib[P]=false;for(r=0;r<t.R.g;r++)t.o[r]=0,t.q[r]=false;},e.Dg=function(t,A){var n,P,r;if(n=e.Zl(t.R,0,A),n>=t.R.f)return  false;if(t.db[n]==1||t.db[n]==2)return  true;if(t.db[n]==3)return  false;if(P=e.Ho(t.R,n),P!=-1)return t.o[P]==1||t.o[P]==2;for(r=0;r<e.$o(t.R,n);r++)if(e.ap(t.R,n,r)==2&&(t.db[e.Zo(t.R,n,r)]==1||t.db[e.Zo(t.R,n,r)]==2))return  true;return  false;},e.Eg=function(t,A){var n;for(n=0;n<e.$o(t.R,A);n++)if(t.o[e._o(t.R,A,n)]==3&&e.ap(t.R,A,n)==1)return  true;return  false;},e.Fg=function(t){var A,n,P,r,i,D,u;for(D=j(l.j7,H,6,t.U+1,15,1),n=0;n<t.R.f;n++)e.Hl(t.R,n)==0&&(t.db[n]==1?++D[t.f[n]]:t.db[n]==2&&--D[t.f[n]]);for(r=0;r<t.R.g;r++)e.fm(t.R,r)==1&&e.cm(t.R,r)==0&&(t.o[r]==1?(++D[t.f[e.Zl(t.R,0,r)]],++D[t.f[e.Zl(t.R,1,r)]]):t.o[r]==2&&(--D[t.f[e.Zl(t.R,0,r)]],--D[t.f[e.Zl(t.R,1,r)]]));for(u=1;u<=t.U;u++)if(D[u]!=0){if(i=D[u]<0,i){for(A=0;A<t.R.f;A++)e.Hl(t.R,A)==0&&(t.db[A]==1?t.db[A]=2:t.db[A]==2&&(t.db[A]=1));for(P=0;P<t.R.g;P++)e.fm(t.R,P)==1&&e.cm(t.R,P)==0&&(t.o[P]==1?t.o[P]=2:t.o[P]==2&&(t.o[P]=1));}return i;}return  false;},e.Gg=function(t,A){var n;return n=j(l.j7,H,6,A,15,1),T.$jb(t,0,n,0,t.length),n;},e.Hg=function(t,A){var n;return n=j(l.Feb,ht,6,A,16,1),T.$jb(t,0,n,0,t.length),n;},e.Ig=function(t){var A,n;if(t.$!=null)for(A=0;A<t.R.f;A++)e.hn(t.R,A,t.$[A]);if(t.i!=null)for(n=0;n<t.R.g;n++)e.Fn(t.R,n,t.i[n]);},e.Jg=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(A=0;A<t.R.f;A++)if(t.db[A]==1||t.db[A]==2){if(o=false,e.Wp(t.R,A))for(u=0;u<e.$o(t.R,A);u++){for(r=e.Zo(t.R,A,u),s=0,L=j(l.j7,H,6,3,15,1),Q=0;Q<e.$o(t.R,r);Q++)L[s]=e.Zo(t.R,r,Q),L[s]!=A&&++s;s==2&&t.f[L[0]]>t.f[L[1]]^L[0]<L[1]&&(o=!o);}else for(u=1;u<e.$o(t.R,A);u++)for(Q=0;Q<u;Q++)i=e.Zo(t.R,A,u),D=e.Zo(t.R,A,Q),t.f[i]>t.f[D]&&(o=!o),i<D&&(o=!o);e.un(t.R,A,t.db[A]==1^o?1:2,t.fb[A]);}else e.un(t.R,A,t.db[A],t.fb[A]);for(n=0;n<t.R.g;n++)if(t.o[n]==1||t.o[n]==2){for(o=false,u=0;u<2;u++)if(P=e.Zl(t.R,u,n),e.$o(t.R,P)==3){for(L=j(l.j7,H,6,2,15,1),s=0,Q=0;Q<3;Q++)e.Zo(t.R,P,Q)!=e.Zl(t.R,1-u,n)&&(L[s++]=e.Zo(t.R,P,Q));t.f[L[0]]>t.f[L[1]]&&(o=!o),L[0]<L[1]&&(o=!o);}e.Jn(t.R,n,t.o[n]==1^o?1:2,t.p[n]);}else e.Jn(t.R,n,t.o[n],t.p[n]);},e.Kg=function(t){var A;for(A=0;A<t.R.f;A++)e.yn(t.R,A,t.M[A]);},e.Lg=function(t){var A,n,P;for(A=0;A<t.R.f;A++)!e.wm(t.R,A)&&t.db[A]==3&&e.ln(t.R,A,true);for(n=0;n<t.R.g;n++)t.o[n]==3&&(P=e.fm(t.R,n),P==1?e.ln(t.R,e.Zl(t.R,0,n),true):P==2&&e.Mn(t.R,n,386));},e.Mg=function(t,A){return t<A?t:t-A;},e.Ng=function(t){e.Og.call(this,t,0);},e.Og=function(t,A){this.R=t,this.Q=A,e.bw(this.R,7),this.c=e.Pg(this.R.f),(this.Q&j1)==0&&e.Cf(this),this.jb=(A&64)!=0||e.vm(this.R),this.b=e.np(this.R)==0,(this.Q&j1)==0&&(this.db=j(l.f7,Y,6,this.R.f,15,1),this.fb=j(l.Feb,ht,6,this.R.f,16,1),this.ib=j(l.Feb,ht,6,this.R.f,16,1),this.o=j(l.f7,Y,6,this.R.g,15,1),this.q=j(l.Feb,ht,6,this.R.g,16,1),this.p=j(l.Feb,ht,6,this.R.g,16,1)),this.d=false,e.Gf(this),(this.Q&j1)==0&&e.Mf(this),e.Lf(this);},e.Pg=function(t){var A;for(A=0;t>0;)t>>=1,++A;return A;},N(34,1,{},e.Ng,e.Og),U.b=false,U.c=0,U.d=false,U.r=false,U.t=0,U.v=0,U.w=0,U.G=false,U.I=0,U.K=false,U.L=false,U.O=0,U.Q=0,U.T=0,U.U=0,U.Z=false,U.jb=false,l.B7=O(34),e.Qg=function(t,A){var n,P,r,i;if(t.d!=A.d)return t.d>A.d?1:-1;for(r=t.a.length,i=A.a.length,n=d.Math.min(r,i),P=0;P<n;P++)if(--r,--i,t.a[r]!=A.a[i])return t.a[r]>A.a[i]?1:-1;return r!=i?r>i?1:-1:t.b!=A.b?t.b>A.b?1:-1:0;},e.Rg=function(t,A){return e.Qg(t,A);},e.Sg=function(){},N(266,1,{},e.Sg),U.Rb=function(t,A){return e.Rg(t,A);},U.mb=function(t){return this===t;},l.s7=O(266),e.Vg=function(){},N(174,1,{174:1},e.Vg),U.b=0,U.c=0,U.d=0,l.r7=O(174),e.Wg=function(t,A){return t.c!=A.c?t.c>A.c?1:-1:0;},e.Xg=function(t,A){return e.Wg(t,A);},e.Yg=function(){},N(267,1,{},e.Yg),U.Rb=function(t,A){return e.Xg(t,A);},U.mb=function(t){return this===t;},l.u7=O(267),e._g=function(){},N(175,1,{175:1},e._g),U.a=0,U.b=0,U.c=0,l.t7=O(175),e.ah=function(t,A,n){t.b==0&&(++t.c,t.b=63),t.b==63?(t.d[t.c]=rt(t.d[t.c],n),t.b-=A):t.b>=A?(t.d[t.c]=h1(t.d[t.c],A),t.d[t.c]=rt(t.d[t.c],n),t.b-=A):(t.d[t.c]=h1(t.d[t.c],t.b),t.d[t.c]=rt(t.d[t.c],t0(n,A-t.b)),A-=t.b,++t.c,t.b=63-A,t.d[t.c]=rt(t.d[t.c],J(n,l.Ueb((1<<A)-1))));},e.bh=function(t,A){t.d[t.c]=m1(t.d[t.c],A);},e.dh=function(t,A){var n;for(n=0;n<t.c;n++)if(pt(t.d[n],A.d[n]))return je(t.d[n],A.d[n])?-1:1;return H1(t.d[t.c],A.d[t.c])?0:je(t.d[t.c],A.d[t.c])?-1:1;},e.eh=function(t,A){t.a=A,t.c=0,t.b=63,y.Oqb(t.d);},e.fh=function(t){this.d=j(l.k7,e0,6,t,14,1);},N(101,1,{101:1,32:1},e.fh),U.Sb=function(t){return e.dh(this,t);},U.a=0,U.b=0,U.c=0,l.v7=O(101),e.hh=function(t,A){return t.b!=A.b?t.b>A.b?-1:1:t.c!=A.c?t.c>A.c?-1:1:0;},e.ih=function(t,A,n){this.b=d.Math.max(t,A),this.c=d.Math.min(t,A),this.a=n;},N(173,1,{173:1,32:1},e.ih),U.Sb=function(t){return e.hh(this,t);},U.a=0,U.b=0,U.c=0,l.w7=O(173),e.kh=function(t,A,n,P){this.a=(w.Mzb(A),y.Lqb(t,j(l.j7,H,6,A,15,1),0,A)),this.b=(w.Mzb(P),y.Lqb(n,j(l.j7,H,6,P,15,1),0,P));},N(208,1,{208:1},e.kh),l.x7=O(208),e.lh=function(t,A){var n,P;for(P=0;P<t.g[A].length;P++)if(n=t.g[A][P],t.f[n]&&(t.o[n]==1||t.o[n]==2)&&t.k[n]==0)return  true;return  false;},e.mh=function(t,A,n){var P,r,i,D,u;for(r=0,D=0,u=0;u<t.g[A].length;u++)P=t.g[A][u],t.k[P]==n&&(i=1<<t.j[P],(D&i)==0&&(D|=i,++r));return r;},e.nh=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h;for(L=new y.Xwb(),s=0;s<t.i.f;s++)if(e.Qo(t.i,s)<2||e.$o(t.i,s)>2)for(D=1;D<e.$o(t.i,s);D++)for(A=e.Zo(t.i,s,D),u=0;u<D;u++)n=e.Zo(t.i,s,u),e.sh(t,A,n)&&(o=e.zh(t,A,n),o&&e.Gr(o,t.f)&&y.Zvb(L.a,o,(T.Qgb(),Fn))==null);for(h=0;h<t.i.g;h++)t.c[h]!=0&&(e.fm(t.i,h)!=2||t.c[h]!=2)||(A=e.Zl(t.i,0,h),n=e.Zl(t.i,1,h),e.sh(t,A,n)&&(o=e.zh(t,A,n),o&&e.Gr(o,t.f)&&y.Zvb(L.a,o,(T.Qgb(),Fn))==null));for(t.g=j(l.j7,E1,7,L.a.c,0,2),t.e=j(l.Feb,ht,6,t.i.f,16,1),i=0,Q=(r=new y.pwb(new y.vwb(new y.Cpb(L.a).a).b),new y.Lpb(r));y.kob(Q.a.a);)for(o=(P=y.nwb(Q.a),P.gd()),t.g[i++]=o.b,D=0;D<o.b.length;D++)t.e[o.b[D]]=true;},e.oh=function(t,A,n,P){var r,i,D,u,o,Q,L;for(r=j(l.j7,H,6,e.$o(t.i,n),15,1),u=0,D=0;D<e.$o(t.i,n);D++)r[u]=e.Zo(t.i,n,D),!P[r[u]]&&e.sh(t,A,r[u])&&++u;if(u==0)return  -1;if(u==1)return r[0];for(o=-1,Q=St,i=0;i<u;i++)L=e.Bp(t.i,A,r[i],St,P),L<Q&&(Q=L,o=r[i]);return o;},e.ph=function(t,A){return t.f[A]&&(t.o[A]==1||t.o[A]==2);},e.qh=function(t,A){return t.e[A];},e.rh=function(t){var A,n;for(n=true,A=0;A<t.i.f;A++)if(t.o[A]!=0&&!t.e[A]){n=false;break;}return n;},e.sh=function(t,A,n){var P,r,i,D,u;if(A==n||t.a[A]!=t.a[n]||t.o[A]!=0&&(t.o[A]==3||t.o[n]==3||t.p[A]^t.o[A]!==t.o[n]||t.k[A]!==t.k[n]||t.j[A]!==t.j[n])||(P=e.Xo(t.i,A,n),P!=-1&&(e.fm(t.i,P)==1&&t.c[P]!=0||e.fm(t.i,P)==2&&t.c[P]==1)))return  false;if(e.Qo(t.i,A)==1&&!e.Tp(t.i,A)){for(r=-1,u=0;u<e.$o(t.i,A);u++)if(e.Zo(t.i,A,u)!=n&&e.ap(t.i,A,u)==2){r=e._o(t.i,A,u);break;}for(i=-1,D=0;D<e.$o(t.i,n);D++)if(e.Zo(t.i,n,D)!=A&&e.ap(t.i,n,D)==2){i=e._o(t.i,n,D);break;}if(r!=-1&&t.c[r]!=0&&t.d[r]^t.c[r]===t.c[i])return  false;}return  true;},e.th=function(t,A,n,P,r){var i,D,u,o,Q,L;for(o=null,i=null,L=0;L<t.g[A].length;L++)D=t.g[A][L],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?i=e.Bh(i,(r[D]<<16)+D):t.k[D]==P&&t.j[D]==n&&(o=e.Bh(o,(r[D]<<16)+D)));if(u=e.Ph(o,i),u==0)return  false;if(u<0)for(Q=0;Q<t.g[A].length;Q++)D=t.g[A][Q],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?(t.k[D]=P<<24>>24,t.j[D]=n<<24>>24):t.k[D]==P&&t.j[D]==n&&(t.k[D]=0,t.j[D]=-1));return  true;},e.uh=function(t,A){var n,P,r,i,D,u;if(!t.b)return  false;for(r=false,i=t.b.a.length-1;i>=0;i--)if(P=false,D=y.xj(t.b,i),D.a==2?P=e.th(t,D.b,D.c,D.d,A):D.a==1&&(P=e.yh(t,D.b,A)),P){for(y.Fj(t.b,D),u=0;u<t.g[D.b].length;u++)n=t.g[D.b][u],t.n[n]=false;r=true;}return r;},e.vh=function(t){var A,n,P,r,i,D,u,o;if(t.g!=null)for(D=new e.Oh(t),t.b=new y.Mj(),r=0;r<t.g.length;r++)P=e.Gh(D,r),P==0?(e.Ch(D,r),u=e.mh(t,r,2),A=e.mh(t,r,1),n=e.lh(t,r),u==1&&A==1&&!n&&(e.xh(t,r,D.a+D.f++),y.sj(t.b,new e.jk(r,1,-1,-1))),u>0?(n&&(e.wh(t,r,D.i+D.g++,2),++u),y.sj(t.b,new e.jk(r,1,-1,-1))):A>0?(n&&e.wh(t,r,D.a+D.f++,1),y.sj(t.b,new e.jk(r,1,-1,-1))):n&&(e.wh(t,r,D.a+D.f++,1),y.sj(t.b,new e.jk(r,1,-1,-1)))):P==1&&(e.lh(t,r)?(i=e.Fh(D,r),o=e.Hh(D,r),y.sj(t.b,new e.jk(r,2,i,o))):(e.Ch(D,r),y.sj(t.b,new e.jk(r,1,-1,-1))));},e.wh=function(t,A,n,P){var r,i;for(i=0;i<t.g[A].length;i++)r=t.g[A][i],t.f[r]&&(t.o[r]==1||t.o[r]==2)&&t.k[r]==0&&(t.k[r]=P<<24>>24,t.j[r]=n<<24>>24);},e.xh=function(t,A,n){var P,r;for(r=0;r<t.g[A].length;r++)P=t.g[A][r],t.k[P]==2&&(t.k[P]=1,t.j[P]=n<<24>>24);},e.yh=function(t,A,n){var P,r,i,D,u,o,Q,L;for(i=t.g[A],r=1,o=0;o<i.length;o++)if(P=i[o],t.f[P]&&t.k[P]==2){r=2;break;}for(D=j(l.j7,E1,7,32,0,2),Q=0;Q<i.length;Q++)P=i[Q],t.f[P]&&t.k[P]==r&&(D[t.j[P]]=e.Bh(D[t.j[P]],(n[P]<<16)+P));for(L=0;L<32;L++)D[L]!=null&&w.vzb(D[L],w.aAb(h0(y.vrb.prototype.kd,y.vrb,[])));if(y.drb(D,new e.Qh()),e.Ph(D[0],D[1])==0)return  false;for(u=0;u<D[0].length;u++)P=D[0][u]&n1,t.k[P]=0,t.j[P]=-1;return  true;},e.zh=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;for(o=j(l.j7,H,6,t.i.f,15,1),k=j(l.j7,H,6,t.i.f,15,1),c=j(l.Feb,ht,6,t.i.f,16,1),Q=j(l.Feb,ht,6,t.i.f,16,1),M=new e.Hr(t.i.f),o[0]=A,k[A]=n,k[n]=-2,M.a[A]=true,M.a[n]=true,i=0,L=0;i<=L;){if(D=o[i],k[D]==D){for(s=0;s<e.$o(t.i,D);s++)if(P=e.Zo(t.i,D,s),!M.a[P]){if(e.ap(t.i,D,s)==2&&e.Tl(t.i,P)<10)o[++L]=P,k[P]=P,Q[P]=Q[D]||e.Qo(t.i,P)==2,c[P]=Q[D]&&!c[D],M.a[P]=true;else if(Q[D]&&c[D]){if(p=e.oh(t,P,k[D],M.a),p==-1)return null;o[++L]=P,k[P]=p,k[p]=-2,Q[P]=false,M.a[P]=true,M.a[p]=true;}else if(e.gq(t.i,e._o(t.i,D,s))&&(o[++L]=P,k[P]=P,Q[P]=false,M.a[P]=true,(e.Tl(t.i,P)==6&&e.Qo(t.i,P)==0||e.Tl(t.i,P)==7&&e.Bl(t.i,P)==1||e.Tl(t.i,P)==14||e.Tl(t.i,P)==15&&e.$o(t.i,P)>2||e.Tl(t.i,P)==16&&e.$o(t.i,P)>2)&&e.$o(t.i,P)>2)){for(u=false,g=1;g<e.$o(t.i,P);g++)if(q=e.Zo(t.i,P,g),!M.a[q])for(B=0;B<g;B++)S=e.Zo(t.i,P,B),M.a[S]||e.sh(t,q,S)&&(o[++L]=q,k[q]=S,k[S]=-2,Q[q]=false,M.a[q]=true,M.a[S]=true,u=true);if(!u)return null;}}}else {for(r=j(l.Feb,ht,6,e.$o(t.i,D),16,1),h=0;h<e.$o(t.i,D);h++)if(P=e.Zo(t.i,D,h),M.a[P])r[h]=k[P]==P;else for(g=0;g<e.$o(t.i,P);g++)if(e.Zo(t.i,P,g)==k[D]){r[h]=true;break;}for(b=0;b<e.$o(t.i,D);b++)if(r[b])if(P=e.Zo(t.i,D,b),M.a[P]){if(e.Xo(t.i,P,k[D])==-1)return null;}else o[++L]=P,k[P]=P,c[P]=false,Q[P]=true,M.a[P]=true;for(s=0;s<e.$o(t.i,D);s++)if(!r[s]&&(P=e.Zo(t.i,D,s),!M.a[P])){if(p=e.oh(t,P,k[D],M.a),p==-1)return null;o[++L]=P,k[P]=p,k[p]=-2,Q[P]=false,M.a[P]=true,M.a[p]=true;}}++i;}return M;},e.Ah=function(t,A,n,P,r,i,D,u,o,Q){this.i=t,this.a=A,this.f=n,this.o=P,this.c=r,this.k=i,this.j=D,this.p=u,this.d=o,this.n=Q,e.nh(this);},e.Bh=function(t,A){var n,P;for(P=j(l.j7,H,6,t==null?1:t.length+1,15,1),n=0;n<P.length-1;n++)P[n]=t[n];return P[P.length-1]=A,P;},N(265,1,{},e.Ah),l.z7=O(265),e.Ch=function(t,A){var n,P,r,i,D,u;for(r=0;r<t.b;r++)if(t.e[r][A]&&t.c[r]!=-3){for(P=0;P<=t.j.g.length;P++)if(P!=A&&t.e[r][P])for(t.e[r][A]=false,u=r<t.a?r:r<t.b?r-t.a:-1,D=e.Mh(t,r<t.a?1:r<t.b?2:0),i=0;i<t.j.g[A].length;i++)n=t.j.g[A][i],e.ph(t.j,n)&&t.j.j[n]==u&&(t.j.j[n]=D<<24>>24);}},e.Dh=function(t,A,n){var P,r,i,D,u;for(u=false,D=1,A[n]=D,P=true;P;){for(P=false,r=0;r<t.b;r++)if(A[r]==D)for(i=0;i<t.b;i++)A[i]==0&&e.Lh(t,r,i)&&(t.c[i]==-2?(A[i]=D+1,P=true):t.c[i]!=t.c[n]&&(A[i]=D+1,u=true));++D;}return u;},e.Eh=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c;for(o=P+1;o<t.j.g.length;o++)if(o!=P&&t.e[A][o]&&t.e[n][o])return D=j(l.j7,H,6,2,15,1),D[0]=n,D[1]=A,D;for(c=j(l.j7,H,6,t.b,15,1),L=j(l.j7,H,6,t.b,15,1),Q=j(l.j7,H,6,t.b,15,1),i=0,s=0,Q[0]=A,L[A]=1;i<=s;){for(h=0;h<t.d[Q[i]].length;h++){if(r=t.d[Q[i]][h],r==n){if(i==0)continue;for(u=L[Q[i]]+1,D=j(l.j7,H,6,u,15,1),D[0]=r,D[1]=Q[i],b=2;b<u;b++)D[b]=c[D[b-1]];return D;}L[r]==0&&t.c[r]!=-3&&(L[r]=L[Q[i]]+1,Q[++s]=r,c[r]=Q[i]);}++i;}return null;},e.Fh=function(t,A){var n;for(n=0;n<t.b;n++)if(t.e[n][A]&&t.c[n]==-3)return n<t.a?n:n<t.b?n-t.a:-1;return  -1;},e.Gh=function(t,A){var n,P;for(n=0,P=0;P<t.b;P++)t.e[P][A]&&t.c[P]==-3&&++n;return n;},e.Hh=function(t,A){var n;for(n=0;n<t.b;n++)if(t.e[n][A]&&t.c[n]==-3)return n<t.a?1:n<t.b?2:0;return  -1;},e.Ih=function(t,A){return A<t.a?A:A<t.b?A-t.a:-1;},e.Jh=function(t,A){return A<t.a?1:A<t.b?2:0;},e.Kh=function(t,A){var n,P;return P=t.j.k[A],n=t.j.j[A],P==0?t.b:P==1?n:t.a+n;},e.Lh=function(t,A,n){var P;for(P=0;P<t.j.g.length;P++)if(t.e[A][P]&&t.e[n][P])return  true;return  false;},e.Mh=function(t,A){return A==1?t.a+t.f++:t.i+t.g++;},e.Nh=function(t,A){var n,P,r,i,D,u,o,Q,L,s;for(L=St,o=-1,s=-1,Q=-1,P=0;P<t.j.i.f;P++)if(e.ph(t.j,P)&&t.j.k[P]!=0)for(u=0;u<A.length;u++)r=e.Ih(t,A[u]),i=e.Jh(t,A[u]),t.j.k[P]==i&&t.j.j[P]==r&&L>t.j.a[P]+(i==1?q1:0)&&(L=t.j.a[P]+(i==1?q1:0),o=r,s=i,Q=A[u]);for(n=0;n<t.j.i.f;n++)e.ph(t.j,n)&&t.j.k[n]==s&&t.j.j[n]==o&&(t.j.k[n]=0,t.j.j[n]=-1);for(D=0;D<t.j.g.length;D++)t.e[Q][D]=false;},e.Oh=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(this.j=t,P=0;P<t.i.f;P++)t.f[P]&&(t.o[P]==1||t.o[P]==2)&&(t.k[P]==1?this.a<=t.j[P]&&(this.a=1+t.j[P]):t.k[P]==2&&this.i<=t.j[P]&&(this.i=1+t.j[P]));for(this.b=this.a+this.i,this.e=y1(l.Feb,[i1,ht],[12,6],16,[this.b+1,t.g.length+1],2),r=0;r<t.i.f;r++)t.f[r]&&(t.o[r]==1||t.o[r]==2)&&!t.e[r]&&(this.e[e.Kh(this,r)][t.g.length]=true);for(o=0;o<t.g.length;o++)for(B=0;B<t.g[o].length;B++)n=t.g[o][B],t.f[n]&&(t.o[n]==1||t.o[n]==2)&&(this.e[e.Kh(this,n)][o]=true);for(this.d=j(l.j7,E1,7,this.b,0,2),Q=0;Q<t.g.length;Q++)for(b=1;b<this.b;b++)if(this.e[b][Q])for(c=0;c<b;c++)this.e[c][Q]&&(this.d[b]=e.Bh(this.d[b],c),this.d[c]=e.Bh(this.d[c],b));for(this.c=j(l.j7,H,6,this.b+1,15,1),h=0;h<this.b;h++)this.e[h][t.g.length]?this.c[h]=-1:this.c[h]=-2;for(L=0;L<t.g.length;L++)if(this.e[this.b][L])for(s=0;s<this.b;s++)this.e[s][L]&&this.c[s]!=L&&(this.c[s]==-2?this.c[s]=L:this.c[s]=-3);for(A=0;A<this.b;A++)if(this.c[A]>=-1&&(i=j(l.j7,H,6,this.b,15,1),e.Dh(this,i,A)))for(s=0;s<this.b;s++)i[s]!=0&&(this.c[s]=-3);for(u=0;u<t.g.length-1;u++)for(b=1;b<this.b;b++)if(this.e[b][u]&&this.c[b]!=-3){for(c=0;c<b;c++)if(this.e[c][u]&&this.c[c]!=-3&&(D=e.Eh(this,b,c,u),D!=null)){for(g=0;g<D.length;g++)this.c[D[g]]=-3;e.Nh(this,D);break;}}},N(359,1,{},e.Oh),U.a=0,U.b=0,U.f=0,U.g=0,U.i=0,l.y7=O(359),e.Ph=function(t,A){var n,P;if(t==null)return A==null?0:1;if(A==null)return  -1;for(n=d.Math.min(t.length,A.length),P=0;P<n;P++)if((t[P]&fA)!=(A[P]&fA))return (t[P]&fA)<(A[P]&fA)?-1:1;return t.length==A.length?0:t.length<A.length?-1:1;},e.Qh=function(){},N(360,1,{},e.Qh),U.Rb=function(t,A){return e.Ph(t,A);},U.mb=function(t){return this===t;},l.A7=O(360),e.Th=function(t,A){switch(A){case 0:return e.Uh(t,false);case 1:return e.Wh(t);case 2:return e.Vh(t);case 3:return e.Yh(t);case 4:return e.Xh(t);}return null;},e.Uh=function(t,A){try{return A?(t=e.fw(t),e.zq(t,!0),e.Tr(t),e.rg(new e.Ng(t))):e.rg(new e.Ng(t));}catch(n){if(n=M1(n),Kt(n,27))return T.Zjb(),null;throw Ut(n);}},e.Vh=function(t){var A,n;try{for(t=e.fw(t),e.pw(t),A=t.r,n=0;n<A;n++)t.H[n]=1,t.T=0;return e.rg(new e.Ng(t));}catch(P){if(P=M1(P),Kt(P,27))return T.Zjb(),null;throw Ut(P);}},e.Wh=function(t){try{return t=e.fw(t),e.pw(t),e.rg(new e.Ng(t));}catch(A){if(A=M1(A),Kt(A,27))return T.Zjb(),null;throw Ut(A);}},e.Xh=function(t){var A;try{return t=e.fw(t),e.pw(t),A=e.Hw(new e.Kw(t)),e.rg(new e.Og(A,8));}catch(n){if(n=M1(n),Kt(n,27))return T.Zjb(),null;throw Ut(n);}},e.Yh=function(t){var A;try{return A=e.Hw(new e.Kw(t)),e.rg(new e.Og(A,8));}catch(n){if(n=M1(n),Kt(n,27))return T.Zjb(),null;throw Ut(n);}},e.Zh=function(t,A,n,P){t.a+=A,t.b+=n,t.c+=P;},e.$h=function(t,A){return t.a+=A.a,t.b+=A.b,t.c+=A.c,t;},e._h=function(t,A){return new e.oi(t.a+A.a,t.b+A.b,t.c+A.c);},e.ai=function(t,A){return t.a=(t.a+A.a)/2,t.b=(t.b+A.b)/2,t.c=(t.c+A.c)/2,t;},e.bi=function(t,A){return t.a!=A.a?t.a<A.a?-1:1:t.b!=A.b?t.b<A.b?-1:1:t.c!=A.c?t.c<A.c?-1:1:0;},e.ci=function(t,A){return new e.oi(t.b*A.c-t.c*A.b,-(t.a*A.c-t.c*A.a),t.a*A.b-t.b*A.a);},e.di=function(t,A){return d.Math.sqrt((A.a-t.a)*(A.a-t.a)+(A.b-t.b)*(A.b-t.b)+(A.c-t.c)*(A.c-t.c));},e.ei=function(t,A){var n;return A==null||!Kt(A,29)?false:(n=A,d.Math.abs(n.a-t.a)+d.Math.abs(n.b-t.b)+d.Math.abs(n.c-t.c)<1e-6);},e.fi=function(t,A){var n,P,r;return P=t.a*t.a+t.b*t.b+t.c*t.c,r=A.a*A.a+A.b*A.b+A.c*A.c,P==0||r==0||(n=(t.a*A.a+t.b*A.b+t.c*A.c)/d.Math.sqrt(P*r),n>=1)?0:n<=-1?A1:d.Math.acos(n);},e.gi=function(t,A){var n,P,r;return n=t.a,P=t.b,r=t.c,t.a=n*A[0][0]+P*A[1][0]+r*A[2][0],t.b=n*A[0][1]+P*A[1][1]+r*A[2][1],t.c=n*A[0][2]+P*A[1][2]+r*A[2][2],t;},e.hi=function(t,A){return t.a*=A,t.b*=A,t.c*=A,t;},e.ii=function(t,A,n,P){t.a=A,t.b=n,t.c=P;},e.ji=function(t,A){return e.ii(t,A.a,A.b,A.c),t;},e.ki=function(t,A){return t.a-=A.a,t.b-=A.b,t.c-=A.c,t;},e.li=function(t,A){return new e.oi(t.a-A.a,t.b-A.b,t.c-A.c);},e.mi=function(t){var A;return A=d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),A==0?(T.Zjb(),t.a=1,t.b=0,t.c=0,t):(t.a/=A,t.b/=A,t.c/=A,t);},e.ni=function(){},e.oi=function(t,A,n){this.a=t,this.b=A,this.c=n;},e.pi=function(t){e.oi.call(this,t.a,t.b,t.c);},N(29,1,{29:1,4:1,32:1},e.ni,e.oi,e.pi),U.Sb=function(t){return e.bi(this,t);},U.mb=function(t){return e.ei(this,t);},U.pb=function(){var t;return t=new l0.Rmb("0.00"),"["+l0.Nmb(t,this.a)+", "+l0.Nmb(t,this.b)+", "+l0.Nmb(t,this.c)+"]";},U.a=0,U.b=0,U.c=0,l.C7=O(29),e.ti=function(t,A){A.Nb(t.c),e.le(A,t.a,t.b);},e.ui=function(t,A){A.c*=t.c,A.a=A.a*t.c+t.a,A.b=A.b*t.c+t.b;},e.vi=function(t,A){e.dn(A,t.c),e._n(A,t.a,t.b);},e.wi=function(t,A){A.a=A.a*t.c+t.a,A.b=A.b*t.c+t.b;},e.xi=function(t,A){A.c=A.c*t.c+t.a,A.d=A.d*t.c+t.b,A.b*=t.c,A.a*=t.c;},e.yi=function(t){t.a=0,t.b=0,t.c=1;},e.zi=function(t){return t.c==1&&t.a==0&&t.b==0;},e.Ai=function(t,A){return A*t.c+t.a;},e.Bi=function(t,A){return A*t.c+t.b;},e.Ci=function(){e.yi(this);},e.Di=function(t,A){this.c=1,this.a=t,this.b=A;},e.Ei=function(t,A,n,P){var r,i,D,u;e.yi(this),A?(P&Yn)==0?Tt.FN(A,t.c,t.d)&&Tt.FN(A,t.c+t.b,t.d+t.a)||((t.b>A.b||t.a>A.a)&&(i=A.b/t.b,u=A.a/t.a,this.c=d.Math.min(i,u)),t.c*this.c<A.c?this.a=A.c-t.c*this.c:(t.c+t.b)*this.c>A.c+A.b&&(this.a=A.c+A.b-(t.c+t.b)*this.c),t.d*this.c<A.d?this.b=A.d-t.d*this.c:(t.d+t.a)*this.c>A.d+A.a&&(this.b=A.d+A.a-(t.d+t.a)*this.c)):(i=A.b/t.b,u=A.a/t.a,D=P&n1,D==0?D=24:(P&n0)!=0&&(D/=256),r=D/n,this.c=d.Math.min(r,d.Math.min(i,u)),this.a=A.c+A.b/2-this.c*(t.c+t.b/2),this.b=A.d+A.a/2-this.c*(t.d+t.a/2)):(P&q1)!=0&&(D=(P&n1)!=0?P&n1:24,this.c=D/n);},N(78,1,{},e.Ci,e.Di,e.Ei),U.pb=function(){return "DepictorTransformation Offset: "+this.a+","+this.b+" Scaling: "+this.c;},U.a=0,U.b=0,U.c=0,l.D7=O(78),e.Gi=function(t){var A,n,P,r;return !T.kjb(w.aAb(t).substr(0,21),ir)||(P=w.aAb(Kn).length,!T.kjb(w.aAb(t).substr(w.aAb(t).length-P,P),Kn))||(n=T.sjb(t,Y1(34),21),n==-1)?null:(r=(w.Wzb(21,n,w.aAb(t).length),w.aAb(t).substr(21,n-21)),A=T.Bjb(t,21+w.aAb(r).length+1,w.aAb(t).length-17),T.kjb(r,"arrow")?new At.HH(A):T.kjb(r,"text")?new e.bx(A):null);},y.Irb=function(t){return new y.Kvb(t);},y.Jrb=function(t){return new dt.dzb(t.Ub());},y.Hi=function(t,A){var n,P,r;for(w.Qzb(A),n=false,r=A.Tb();r.dd();)P=r.ed(),n=n|t.add(P);return n;},y.Ii=function(t,A,n){var P,r;for(r=t.Tb();r.dd();)if(P=r.ed(),F1(A)===F1(P)||A!=null&&T.Oc(A,P))return n&&r.fd(),true;return  false;},y.Ji=function(t,A){var n,P;for(w.Qzb(A),P=A.Tb();P.dd();)if(n=P.ed(),!t.contains(n))return  false;return  true;},y.Ki=function(t,A){var n,P,r;for(w.Qzb(A),n=false,P=t.Tb();P.dd();)r=P.ed(),A.contains(r)&&(P.fd(),n=true);return n;},y.Li=function(t,A){var n,P,r;for(w.Qzb(A),n=false,P=t.Tb();P.dd();)r=P.ed(),A.contains(r)||(P.fd(),n=true);return n;},y.Mi=function(t,A){var n,P,r,i;for(i=t.size(),A.length<i&&(A=w.wzb(new Array(i),A)),r=A,P=t.Tb(),n=0;n<i;++n)$1(r,n,P.ed());return A.length>i&&l.U5(A,i,null),A;},y.Ni=function(t){var A,n,P;for(P=new y.Pvb(", ","[","]"),n=t.Tb();n.dd();)A=n.ed(),y.Mvb(P,F1(A)===F1(t)?"(this Collection)":A==null?rA:Ye(A));return P.a?w.aAb(P.e).length==0?P.a.a:P.a.a+(""+P.e):P.c;},N(43,1,_n),U.Ub=function(){return new y.Kvb(this);},U.Vb=function(){return new dt.dzb(this.Ub());},U.add=function(t){throw Ut(new T.bkb("Add not supported on this collection"));},U.addAll=function(t){return y.Hi(this,t);},U.clear=function(){var t;for(t=this.Tb();t.dd();)t.ed(),t.fd();},U.contains=function(t){return y.Ii(this,t,false);},U.containsAll=function(t){return y.Ji(this,t);},U.isEmpty=function(){return this.size()==0;},U.remove=function(t){return y.Ii(this,t,true);},U.removeAll=function(t){return y.Ki(this,t);},U.retainAll=function(t){return y.Li(this,t);},U.toArray=function(){return this.Wb(j(l.Sbb,C1,1,this.size(),5,1));},U.Wb=function(t){return y.Mi(this,t);},U.pb=function(){return y.Ni(this);},l.gcb=O(43),y.Cub=function(t){return new y.Kvb(t);},y.Dub=function(t){var A;return dt.bzb(t.Vb(),(A=dt.lyb(new dt.syb(),dt._xb(new dt.qyb(),new dt.oyb(),new dt.Syb(),v(F(l.ydb,1),s0,96,0,[(dt.dyb(),dt.byb)]))),dt.hyb(A,new dt.vyb())));},y.aj=function(t,A){var n,P;for(n=0,P=t.size();n<P;++n)if(y.Rub(A,t.getAtIndex(n)))return n;return  -1;},y.bj=function(t,A,n){return new y.vob(t,A,n);},N(95,43,G8),U.Ub=function(){return new y.Kvb(this);},U.addAtIndex=function(t,A){throw Ut(new T.bkb("Add not supported on this list"));},U.add=function(t){return this.addAtIndex(this.size(),t),true;},U.addAllAtIndex=function(t,A){var n,P,r;for(w.Qzb(A),n=false,r=A.Tb();r.dd();)P=r.ed(),this.addAtIndex(t++,P),n=true;return n;},U.clear=function(){this.Xb(0,this.size());},U.mb=function(t){var A,n,P,r,i;if(t===this)return  true;if(!Kt(t,83)||(i=t,this.size()!=i.size()))return  false;for(r=i.Tb(),n=this.Tb();n.dd();)if(A=n.ed(),P=r.ed(),!(F1(A)===F1(P)||A!=null&&T.Oc(A,P)))return  false;return  true;},U.ob=function(){return y.Lrb(this);},U.indexOf=function(t){return y.aj(this,t);},U.Tb=function(){return new y.nob(this);},U.lastIndexOf=function(t){var A;for(A=this.size()-1;A>-1;--A)if(y.Rub(t,this.getAtIndex(A)))return A;return  -1;},U.removeAtIndex=function(t){throw Ut(new T.bkb("Remove not supported on this list"));},U.Xb=function(t,A){var n,P;for(P=new y.tob(this,t),n=t;n<A;++n)w.Ozb(P.a<P.c.size()),P.c.getAtIndex(P.b=P.a++),y.mob(P);},U.setAtIndex=function(t,A){throw Ut(new T.bkb("Set not supported on this list"));},U.subList=function(t,A){return y.bj(this,t,A);},l.ncb=O(95);function Yu(f){return f==null?false:f.$implements__java_lang_Cloneable||Array.isArray(f);}y.qj=function(t){t.a=j(l.Sbb,C1,1,0,5,1);},y.rj=function(t,A,n){w.Szb(A,t.a.length),w.pzb(t.a,A,n);},y.sj=function(t,A){return w.szb(t.a,A),true;},y.tj=function(t,A,n){var P,r;return w.Szb(A,t.a.length),P=n.toArray(),r=P.length,r==0?false:(w.qzb(t.a,A,P),true);},y.uj=function(t,A){var n,P;return n=A.toArray(),P=n.length,P==0?false:(w.qzb(t.a,t.a.length,n),true);},y.vj=function(t){w.uzb(t.a,0);},y.wj=function(t,A){return y.zj(t,A,0)!=-1;},y.xj=function(t,A){return w.Pzb(A,t.a.length),t.a[A];},y.yj=function(t,A){return y.zj(t,A,0);},y.zj=function(t,A,n){for(;n<t.a.length;++n)if(y.Rub(A,t.a[n]))return n;return  -1;},y.Aj=function(t){return t.a.length==0;},y.Bj=function(t){return new y.Bqb(t);},y.Cj=function(t,A){return y.Dj(t,A,t.a.length-1);},y.Dj=function(t,A,n){for(;n>=0;--n)if(y.Rub(A,t.a[n]))return n;return  -1;},y.Ej=function(t,A){var n;return n=(w.Pzb(A,t.a.length),t.a[A]),w.tzb(t.a,A,1),n;},y.Fj=function(t,A){var n;return n=y.zj(t,A,0),n==-1?false:(y.Ej(t,n),true);},y.Gj=function(t,A,n){var P;w.Tzb(A,n,t.a.length),P=n-A,w.tzb(t.a,A,P);},y.Hj=function(t,A,n){var P;return P=(w.Pzb(A,t.a.length),t.a[A]),t.a[A]=n,P;},y.Ij=function(t){return t.a.length;},y.Jj=function(t,A){y.crb(t.a,t.a.length,A);},y.Kj=function(t){return w.mzb(t.a);},y.Lj=function(t,A){var n,P;for(P=t.a.length,A.length<P&&(A=w.wzb(new Array(P),A)),n=0;n<P;++n)$1(A,n,t.a[n]);return A.length>P&&l.U5(A,P,null),A;},y.Mj=function(){y.qj(this);},y.Nj=function(t){y.qj(this),w.Kzb(t>=0,"Initial capacity must not be negative");},N(20,95,Wn,y.Mj,y.Nj),U.addAtIndex=function(t,A){y.rj(this,t,A);},U.add=function(t){return y.sj(this,t);},U.addAllAtIndex=function(t,A){return y.tj(this,t,A);},U.addAll=function(t){return y.uj(this,t);},U.clear=function(){y.vj(this);},U.contains=function(t){return y.wj(this,t);},U.getAtIndex=function(t){return y.xj(this,t);},U.indexOf=function(t){return y.yj(this,t);},U.isEmpty=function(){return y.Aj(this);},U.Tb=function(){return y.Bj(this);},U.lastIndexOf=function(t){return y.Cj(this,t);},U.removeAtIndex=function(t){return y.Ej(this,t);},U.remove=function(t){return y.Fj(this,t);},U.Xb=function(t,A){y.Gj(this,t,A);},U.setAtIndex=function(t,A){return y.Hj(this,t,A);},U.size=function(){return y.Ij(this);},U.toArray=function(){return y.Kj(this);},U.Wb=function(t){return y.Lj(this,t);},l.Fcb=O(20),e.ek=function(t){var A,n,P;for(P=new T.Ujb(),n=new y.Bqb(t);n.a<n.c.a.length;)A=y.Aqb(n),T.Sjb(T.Sjb(P,ir+A.Lb()+'"'+A.Kb()+Kn),k1);return P.a;},e.fk=function(){y.Mj.call(this);},e.gk=function(t){var A,n;y.Mj.call(this);try{if(t)for(n=new y.Bqb(t);n.a<n.c.a.length;)A=y.Aqb(n),y.sj(this,A.Gb());}catch(P){if(P=M1(P),!Kt(P,19))throw Ut(P);}},e.hk=function(t){var A,n,P;if(y.Mj.call(this),!(t==null||w.aAb(t).length==0))for(A=0,n=T.rjb(t,Y1(10));n!=-1;)P=e.Gi((w.Wzb(A,n,w.aAb(t).length),w.aAb(t).substr(A,n-A))),P&&w.szb(this.a,P),A=n+1,n=T.sjb(t,Y1(10),A);},N(103,20,Wn,e.fk,e.gk,e.hk),U.pb=function(){return e.ek(this);},l.E7=O(103),e.jk=function(t,A,n,P){this.b=t,this.a=A,this.c=n,this.d=P;},N(109,1,{109:1},e.jk),U.a=0,U.b=0,U.c=0,U.d=0,l.F7=O(109),e.kk=function(t){var A,n,P,r,i,D;if(t.j!=0)return t.j;if(t.i&&e.Tl(t.e,t.a)!=15&&e.Tl(t.e,t.a)!=16){for(D=0;D<e.Lo(t.e,t.a);D++)if(i=e._o(t.e,t.a,D),e.Sm(t.e,i,t.a))return e.Zo(t.e,t.a,D)==t.b?t.j=e.im(t.e,i)==257?3:1:t.j=e.im(t.e,i)==257?1:3,t.j;}return A=e.Yl(t.e,t.a,t.g),P=e.Yl(t.e,t.a,t.b),P<A&&(P+=I1),e.Lo(t.e,t.a)==2?(n=P-A,n>3.0915926535897933&&n<3.191592653589793?(t.j=-1,t.j):(t.j=n<A1?4:2,t.j)):(r=e.Yl(t.e,t.a,t.d),r<A&&(r+=I1),t.j=r<P?2:4,t.j);},e.lk=function(t,A,n,P){var r,i,D,u;for(this.e=t,this.g=n,this.a=P,D=-1,u=0;u<e.Lo(this.e,this.a);u++){if(r=e.Zo(this.e,this.a,u),i=e._o(this.e,this.a,u),r==this.g){e.im(this.e,i)==386&&(this.j=-1);continue;}e.Sm(this.e,i,this.a)&&(this.i&&(t.u[P]|=q1),this.i=true),D==A[r]?(this.d=r,this.f=true,this.c=e.gq(this.e,i)):D<A[r]?(D=A[r],this.d=this.b,this.b=r):this.d=r;}},N(116,1,{},e.lk),U.a=0,U.b=0,U.c=false,U.d=0,U.f=false,U.g=0,U.i=false,U.j=0,l.G7=O(116),e.mk=function(t,A,n,P){this.c=t,this.b=A,this.d=n,this.a=P;},N(18,1,{18:1},e.mk),U.pb=function(){return this.d;},U.a=0,U.c=0,l.H7=O(18),e.ok=function(t){var A,n,P,r,i;if(r=new y.cvb(new e.Hk()),i=new y._ub(),t.k!=null)for(P=0;P<t.k.length;P++)for(A=0;A<t.k[P].r;A++)y.Xub(r,w.aAb(t.e[P].R.c*e.dm(t.k[P],A))),y.Xub(i,y.Yub(r)),r.b.a.length<i.b.a.length&&y.Xub(r,y.Yub(i));return n=r.b.a.length+i.b.a.length,n==0?e.pk(t):t.p.c*(n%2==0?(T._hb(r.b.a.length==0?null:y.xj(r.b,0))+T._hb(i.b.a.length==0?null:y.xj(i.b,0)))/2:T._hb(r.b.a.length==0?null:y.xj(r.b,0)));},e.pk=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g;if(s=U0,h=or,c=U0,g=or,r=0,t.k!=null)for(u=0;u<t.k.length;u++)for(n=0;n<t.k[u].q;n++)L=e.Ai(t.e[u].R,e.Dl(t.k[u],n).a),b=e.Bi(t.e[u].R,e.Dl(t.k[u],n).b),s=d.Math.min(s,L),h=d.Math.max(h,L),c=d.Math.min(c,b),g=d.Math.max(g,b),++r;return r<=1?t.d:(i=h-s,D=g-c,o=(i+D)/2,Q=3*r,A=i*D,P=o/(1-Q),d.Math.sqrt(P*P-A/(1-Q))-P);},e.qk=function(t){var A;if(t.p=new e.Ci(),t.k!=null)for(t.e=j(l.aab,C1,105,t.k.length,0,1),A=0;A<t.k.length;A++)t.e[A]=new Tt.bN(t.k[A]);if(t.a!=null)for(t.b=j(l.aab,C1,105,t.a.length,0,1),A=0;A<t.a.length;A++)t.b[A]=new Tt.bN(t.a[A]);t.d=24;},e.rk=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;for(r=j(l.hab,C1,35,t.k.length,0,1),S=0,q=0,b=0;b<t.k.length;b++)e.be(t.e[b],A,null,q1),r[b]=t.e[b].v,S+=r[b].b,q=d.Math.max(q,r[b].a);for(u=j(l.hab,C1,35,t.a.length,0,1),p=0,M=0,c=0;c<t.a.length;c++)e.be(t.b[c],A,null,65552),u[c]=t.b[c].v,p=d.Math.max(p,u[c].b),M+=u[c].a+12;if(P=d.Math.max(48,p+24),q=d.Math.max(q,24+2*M),n=-1,!t.g)t.g=new e.fk(),y.sj(t.g,new At.GH()),n=0;else {for(g=0;g<t.g.a.length;g++)if(Kt(y.xj(t.g,g),61)){n=g;break;}n==-1&&(n=t.g.a.length,y.sj(t.g,new At.GH()));}for(k=18,h=0;h<t.k.length;h++){if(h==t.n){for(At.FH(y.xj(t.g,n),k-18,q/2,k-18+P,q/2),i=k+-6,D=.5*(q-12)-M,B=0;B<t.a.length;B++)Q=i+.5*(p-u[B].b)-u[B].c,s=D-u[B].d,e.dd(t.b[B],new e.Di(Q,s)),D+=12+u[B].a;k+=P;}o=k-r[h].c,L=.5*(q-r[h].a)-r[h].d,e.dd(t.e[h],new e.Di(o,L)),k+=36+r[h].b;}t.o=false;},e.sk=function(t,A){var n,P;if(t.g)for(P=new y.Bqb(t.g);P.a<P.c.a.length;)n=y.Aqb(P),n.Ib(A,t.p);},e.tk=function(t,A){var n,P,r,i,D;if(t.i!=0&&t.k!=null){for(P=e.ok(t)/t.p.c,C.$2(A,t.i),C.Y2(A,Rt(1.6*P),true,false),i=0;i<t.k.length;i++)if(t.k[i].q!=0){for(r=new v1.Xfb(),n=0;n<t.k[i].q;n++)r.a=Rt(r.a+e.Ql(t.k[i],n)),r.b=Rt(r.b+e.Rl(t.k[i],n));r.a=r.a/t.k[i].q|0,r.b=r.b/t.k[i].q|0,r.a=Rt(e.Ai(t.e[i].R,r.a)),r.b=Rt(e.Bi(t.e[i].R,r.b)),D=t.c==0?""+(i+1):t.c==2?i<t.j?""+String.fromCharCode(65+i&n1):"R"+(i+1-t.j):t.c==1?i<t.n?""+String.fromCharCode(65+i&n1):"P"+(i+1-t.n):"?"+(i+1),C.H2(A,r.a,r.b,D);}}},e.uk=function(t,A){var n,P,r,i,D;if(t.e!=null)for(n=e.ok(t)/t.p.c,r=t.e,i=0,D=r.length;i<D;++i)P=r[i],e.Wd(P,t.f),P.r=n/P.R.c,e.Td(P,A);if(t.b!=null)for(r=t.b,i=0,D=r.length;i<D;++i)P=r[i],e.Td(P,A);},e.vk=function(t,A){t.d=A;},e.wk=function(t,A){t.f=A;},e.xk=function(t,A){var n,P,r,i;if(t.e!=null)for(P=t.e,r=0,i=P.length;r<i;++r)n=P[r],n.J=A;},e.yk=function(t,A,n){var P,r,i,D,u,o,Q;if(t.e!=null)for(i=t.e,u=0,Q=i.length;u<Q;++u)P=i[u],e.Xd(P,A,n);if(t.b!=null)for(r=t.b,D=0,o=r.length;D<o;++D)P=r[D],e.Xd(P,A,n);},e.zk=function(t,A){t.i=A;},e.Ak=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k;if(e.Bk(t,A,n,P),e.zi(t.p))return null;if(t.k!=null)for(h=t.k,b=0,c=h.length;b<c;++b)s=h[b],e.vi(t.p,s);if(t.g)for(B=new y.Bqb(t.g);B.a<B.c.a.length;)g=y.Aqb(B),e.ti(t.p,g);if(t.e!=null)for(D=t.e,o=0,L=D.length;o<L;++o)r=D[o],e.yi(r.R);if(t.b!=null)for(i=t.b,u=0,Q=i.length;u<Q;++u)r=i[u],e.yi(r.R);return k=t.p,t.p=new e.Ci(),k;},e.Bk=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B;if(t.o&&e.rk(t,A),i=null,t.e!=null)for(u=t.e,Q=0,s=u.length;Q<s;++Q)D=u[Q],e.be(D,A,null,0),i=i?Tt.IN(i,D.v):D.v;if(t.b!=null)for(u=t.b,Q=0,s=u.length;Q<s;++Q)D=u[Q],e.be(D,A,null,0),i=i?Tt.IN(i,D.v):D.v;if(t.g)for(c=new y.Bqb(t.g);c.a<c.c.a.length;)b=y.Aqb(c),g=b.Jb(A),e.xi(t.p,g),i=i?Tt.IN(i,g):g;if(!i)return null;if(r=e.ok(t),B=new e.Ei(i,n,r,P),!(B.c==1&&B.a==0&&B.b==0)){if(e.ui(B,t.p),t.e!=null)for(o=t.e,L=0,h=o.length;L<h;++L)D=o[L],e.dd(D,B);if(t.b!=null)for(u=t.b,Q=0,s=u.length;Q<s;++Q)D=u[Q],e.dd(D,B);return B;}return null;},e.Ck=function(t,A){t&&(this.k=j(l.j8,OA,26,1,0,1),this.k[0]=t),this.c=0,this.g=A,e.qk(this);},e.Dk=function(t,A){var n,P;for(this.k=j(l.j8,OA,26,t.g.a.length+t.f.a.length,0,1),P=0;P<t.g.a.length+t.f.a.length;P++)this.k[P]=P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length);for(this.n=t.g.a.length,this.a=j(l.j8,OA,26,t.a.a.length,0,1),n=0;n<t.a.a.length;n++)this.a[n]=y.xj(t.a,n);this.o=false,this.c=1,this.g=A,e.qk(this);},e.Ek=function(t,A,n){this.k=t,this.c=2,this.g=n,this.j=A,e.qk(this);},e.Fk=function(t,A){this.k=t,this.c=0,this.g=A,e.qk(this);},e.Gk=function(t,A){return w.cAb((w.Qzb(t),t))>w.cAb((w.Qzb(A),A))?-1:w.cAb((w.Qzb(t),t))<w.cAb((w.Qzb(A),A))?1:0;},N(158,1,{},e.Ck,e.Dk,e.Ek,e.Fk),U.c=0,U.d=0,U.f=0,U.i=0,U.j=0,U.n=0,U.o=false,l.J7=O(158),e.Hk=function(){},N(358,1,{},e.Hk),U.Rb=function(t,A){return e.Gk(t,A);},U.mb=function(t){return this===t;},l.I7=O(358),e.Rk=function(){e.Rk=D1,e.Lk=v(F(l.Ybb,1),st,2,6,["?","H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr","Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"R4","R5","R6","R7","R8","R9","R10","R11","R12","R13","R14","R15","R16","R1","R2","R3","A","A1","A2","A3",R1,R1,"D","T","X","R","H2","H+","Nnn","HYD","Pol",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"Ala","Arg","Asn","Asp","Cys","Gln","Glu","Gly","His","Ile","Leu","Lys","Met","Phe","Pro","Ser","Thr","Trp","Tyr","Val"]),e.Pk=v(F(l.Eeb,1),Yt,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]),e.Ok=v(F(l.f7,1),Y,6,15,[6]),e.Kk=v(F(l.f7,1),Y,6,15,[2]),e.Mk=v(F(l.f7,2),bA,10,0,[null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[0]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[0]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[2,4,6]),v(F(l.f7,1),Y,6,15,[1,3,5,7]),v(F(l.f7,1),Y,6,15,[0]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[2,4,6]),v(F(l.f7,1),Y,6,15,[1,3,5,7]),v(F(l.f7,1),Y,6,15,[0,2]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,v(F(l.f7,1),Y,6,15,[1,2,3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[2,4,6]),v(F(l.f7,1),Y,6,15,[1,3,5,7]),v(F(l.f7,1),Y,6,15,[0,2,4,6]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2])]),e.Nk=v(F(l.f7,2),bA,10,0,[null,v(F(l.f7,1),Y,6,15,[1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,v(F(l.f7,1),Y,6,15,[-3]),v(F(l.f7,1),Y,6,15,[-2]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),null,v(F(l.f7,1),Y,6,15,[-3]),v(F(l.f7,1),Y,6,15,[-2]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3,4]),v(F(l.f7,1),Y,6,15,[2,3,4,5]),v(F(l.f7,1),Y,6,15,[2,3,6]),v(F(l.f7,1),Y,6,15,[2,3,4,7]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[1,2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[-3,3,5]),v(F(l.f7,1),Y,6,15,[-2]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[6]),v(F(l.f7,1),Y,6,15,[4,6,7]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[-3,3,5]),v(F(l.f7,1),Y,6,15,[-2,4,6]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[5]),v(F(l.f7,1),Y,6,15,[6]),v(F(l.f7,1),Y,6,15,[4,6,7]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[1,3]),v(F(l.f7,1),Y,6,15,[1,2]),v(F(l.f7,1),Y,6,15,[1,3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[-2,2,4]),v(F(l.f7,1),Y,6,15,[-1,1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[4,5]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[3])]);},e.Sk=function(t,A){var n,P;n=-1,P=0,(t.G[A]&16)!=0&&(n=16,++P),(t.G[A]&64)!=0&&(n=64,++P),(t.G[A]&32)!=0&&(n=32,++P),(t.G[A]&4)!=0&&(n=4,++P),(t.G[A]&2)!=0&&(n=2,++P),(t.G[A]&8)!=0&&(n=8,++P),(t.G[A]&1)!=0&&(n=1,++P),n!=-1&&n!=(t.H[A]&127)&&(t.H[A]=n),P<2&&(t.G[A]&=-128);},e.Tk=function(t,A,n,P){var r;return r=e.Uk(t,6),e.ii(t.J[r],A,n,P),r;},e.Uk=function(t,A){return t.q>=t.M&&e.Rn(t,t.M*2),t.C[t.q]=0,e.Cn(t,t.q,A),t.s[t.q]=0,t.u[t.q]=0,t.B[t.q]=0,t.w[t.q]=0,e.ii(t.J[t.q],0,0,0),t.v!=null&&(t.v[t.q]=null),t.t!=null&&(t.t[t.q]=null),t.T=0,t.q++;},e.Vk=function(t,A,n){return e.Wk(t,A,n,e.Lm(t,A)||e.Lm(t,n)?16:1);},e.Wk=function(t,A,n,P){var r;if(A==n)return  -1;for(r=0;r<t.r;r++)if(t.D[0][r]==A&&t.D[1][r]==n||t.D[0][r]==n&&t.D[1][r]==A)return t.H[r]<P&&(t.H[r]=P),r;return e.Xk(t,A,n,P);},e.Xk=function(t,A,n,P){return t.r>=t.N&&e.Sn(t,t.N*2),t.D[0][t.r]=A,t.D[1][t.r]=n,t.H[t.r]=P,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},e.Yk=function(t,A){return e.Zk(t,A,A.q,A.r,true);},e.Zk=function(t,A,n,P,r){var i,D,u,o,Q;for(t.K=t.K|A.K,D=j(l.j7,H,6,A.q,15,1),o=e.cn(t,1),Q=e.cn(t,2),i=0;i<n;i++)(r||H1(J(A.B[i],r0),0))&&(D[i]=e.ll(A,t,i,o,Q));for(u=0;u<P;u++)(r||!e.Jm(A,A.D[0][u])&&!e.Jm(A,A.D[1][u]))&&e.ml(A,t,u,o,Q,D[A.D[0][u]],D[A.D[1][u]],false);return t.L=t.L&&A.L,t.I=0,t.T=0,D;},e.$k=function(t,A,n,P,r,i,D,u){var o,Q;return o=e.xl(t,A,n),o==-1?(t.q>=t.M&&e.Rn(t,t.M*2),o=e.Uk(t,P),e.ii(t.J[o],A,n,0),t.A[o]=r,e.gn(t,o,i),e.wn(t,o,D),e.mn(t,o,u),true):(Q=e.fl(t,o,P,r,i,D),Q=Q|e.mn(t,o,u),Q);},e._k=function(t,A,n,P){var r;for(r=0;r<t.r;r++)if(t.D[0][r]==A&&t.D[1][r]==n||t.D[0][r]==n&&t.D[1][r]==A)return e.il(t,r,P),t.T=0,r;return t.r>=t.N&&e.Sn(t,t.N*2),t.D[0][t.r]=A,t.D[1][t.r]=n,t.H[t.r]=P,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},e.al=function(t,A,n,P,r,i){for(var D,u,o;t.q+P>t.M;)e.Rn(t,t.M*2);for(;t.r+P>t.N;)e.Sn(t,t.N*2);return D=e.xl(t,A,n),D!=-1?e.bl(t,D,P,r,i):(u=e.yl(t,A,n),u!=-1?e.cl(t,u,P,r,i):(D=e.Tk(t,A,n,0),o=A1*(P-2)/P,e.Wm(t,D,P,D,r,0,A1-o,i),t.T=0,true));},e.bl=function(t,A,n,P,r){var i,D,u,o,Q,L;if(P&&e.vp(t,A)>1||!P&&e.vp(t,A)>2)return  false;for(D=0,i=j(l.h7,Ft,6,4,15,1),o=0;o<t.r;o++){for(Q=0;Q<2;Q++)if(t.D[Q][o]==A){if(D==2){D=3;break;}i[D++]=e.Yl(t,A,t.D[1-Q][o]);}if(D==3)break;}return D==3?false:(L=D==1?i[0]+A1:d.Math.abs(i[0]-i[1])>A1?(i[0]+i[1])/2:(i[0]+i[1])/2+A1,u=A1*(n-2)/n,e.Wm(t,A,n,A,P,L-u/2,A1-u,r),t.T=0,true);},e.cl=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c,g;if(Q=j(l.j7,H,6,2,15,1),o=j(l.h7,Ft,6,2,15,1),Q[0]=t.D[0][A],Q[1]=t.D[1][A],e.vp(t,Q[0])>3||e.vp(t,Q[1])>3)return  false;for(D=0,i=j(l.h7,Ft,6,4,15,1),h=0;h<t.r;h++)if(h!=A){for(b=0;b<2;b++){for(c=0;c<2;c++)if(t.D[b][h]==Q[c]){if(D==4){D=5;break;}i[D++]=e.Yl(t,Q[c],t.D[1-b][h]);}if(D==5)break;}if(D==5)break;}if(D==5)return  false;for(o[0]=e.Yl(t,Q[0],Q[1]),o[0]<0?(o[1]=o[0]+A1,u=0):(o[1]=o[0],o[0]=o[1]-A1,u=1),g=0,s=0;s<D;s++)i[s]>o[0]&&i[s]<o[1]?--g:++g;return u=g>0?1-u:u,L=A1*(n-2)/n,e.Wm(t,Q[u],n-1,Q[1-u],P,o[g>0?0:1]+A1-L,A1-L,r),t.T=0,true;},e.dl=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g;for(r=j(l.j7,H,6,A.q,15,1),o=e.cn(t,1),Q=e.cn(t,2),P=0;P<A.q;P++)if(A.C[P]!=0||(A.t==null||A.t[P]==null?null:T.ijb((b=A.t[P],S1.Mmb(),b)))!=null)r[P]=e.ll(A,t,P,o,Q);else if(r[P]=n,(A.T&8)!=0){for(u=0;u<A.r;u++)for(L=0;L<2;L++)if(A.D[L][u]==P){if(c=A.D[1-L][u],g=A.u[c]&3,s=false,g==1||g==2){for(i=0;i<A.r;i++)if(i!=u)for(h=0;h<2;h++)A.D[h][i]==c&&A.D[1-h][i]<P&&(s=!s);}s&&e.un(A,c,g==1?2:1,(A.u[c]&4)!=0);}}for(D=0;D<A.r;D++)e.ml(A,t,D,o,Q,r[A.D[0][D]],r[A.D[1][D]],false);return t.L=t.L&&A.L,t.I=0,t.T=0,r;},e.el=function(t,A){var n,P,r,i,D,u,o,Q,L;return n=t.J[A[0]],P=t.J[A[1]],r=t.J[A[2]],i=t.J[A[3]],o=new e.oi(P.a-n.a,P.b-n.b,P.c-n.c),Q=new e.oi(r.a-P.a,r.b-P.b,r.c-P.c),L=new e.oi(i.a-r.a,i.b-r.b,i.c-r.c),D=new e.oi(o.b*Q.c-o.c*Q.b,-(o.a*Q.c-o.c*Q.a),o.a*Q.b-o.b*Q.a),u=new e.oi(Q.b*L.c-Q.c*L.b,-(Q.a*L.c-Q.c*L.a),Q.a*L.b-Q.b*L.a),-d.Math.atan2(d.Math.sqrt(Q.a*Q.a+Q.b*Q.b+Q.c*Q.c)*(o.a*u.a+o.b*u.b+o.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},e.fl=function(t,A,n,P,r,i){return (n==1||n==151||n==152)&&e.vp(t,A)>1||(t.B[A]=J(t.B[A],-2),t.v!=null&&(t.v[A]=null),t.t!=null&&(t.t[A]=null),n==t.C[A]&&P==t.A[A]&&r==((t.u[A]&cA)>>>27)-1&&i==(t.u[A]&48))?false:((n==151||n==152)&&(P=n-149,n=1),t.u[A]&=960,t.C[A]=n,t.A[A]=P,t.s[A]=0,t.B[A]=0,e.gn(t,A,r),e.wn(t,A,i),e.an(t,t.w[A]),t.T=0,true);},e.gl=function(t,A,n,P){var r;return r=e.xl(t,A,n),r!=-1&&e.hl(t,r,P);},e.hl=function(t,A,n){if(n){if(t.s[A]>8)return  false;++t.s[A];}else {if(t.s[A]<-8)return  false;--t.s[A];}return t.T=0,true;},e.il=function(t,A,n){var P,r,i,D,u;return i=false,D=t.H[A],n==511?i=e.sm(t,A):e.Eq(t,A,n)&&(n==257||n==129?(P=e.Xm(t,A,t.D[0][A]),r=e.Xm(t,A,t.D[1][A]),n==D?(P==r||r)&&(u=t.D[0][A],t.D[0][A]=t.D[1][A],t.D[1][A]=u,i=true):(!P&&r&&(u=t.D[0][A],t.D[0][A]=t.D[1][A],t.D[1][A]=u),t.H[A]=n,i=true)):(t.H[A]=n,i=true)),i&&(t.T=(D&127)==(n&127)?t.T&7:0,t.G[A]=0),i;},e.jl=function(t){t.q=0,t.r=0,t.K=false,t.L=false,t.I=0,t.v=null,t.t=null,t.P=null,t.T=0;},e.kl=function(t){var A,n,P,r,i,D,u,o;for(D=0;D<t.r;D++)t.H[D]==512&&(n=t.D[0][D],P=t.D[1][D],t.C[n]==-1^t.C[P]==-1&&t.s[n]!=0&&t.s[P]!=0&&t.s[n]<0^t.s[P]<0&&(t.s[n]<0?(++t.s[n],--t.s[P]):(--t.s[n],++t.s[P])));for(o=j(l.j7,H,6,t.q,15,1),r=0,A=0;A<t.q;A++){if(t.C[A]==-1){o[A]=-1;continue;}r<A&&(t.C[r]=t.C[A],t.s[r]=t.s[A],t.A[r]=t.A[A],t.u[r]=t.u[A],t.B[r]=t.B[A],t.w[r]=t.w[A],e.ji(t.J[r],t.J[A]),t.v!=null&&(t.v[r]=t.v[A]),t.t!=null&&(t.t[r]=t.t[A])),o[A]=r,++r;}for(t.q=r,u=0,i=0;i<t.r;i++)t.H[i]!=512&&(t.H[u]=t.H[i],t.F[u]=t.F[i],t.G[u]=t.G[i],t.D[0][u]=o[t.D[0][i]],t.D[1][u]=o[t.D[1][i]],++u);return t.r=u,o;},e.ll=function(t,A,n,P,r){var i,D,u;return i=A.q,i>=A.M&&e.Rn(A,A.M*2),u=(t.u[n]&Q1)>>18,D=-1,u==1?P==-1?D=e.cn(A,u):D=d.Math.min(31,P+((t.u[n]&Q1)>>18!=1&&(t.u[n]&Q1)>>18!=2?-1:(t.u[n]&F0)>>20)):u==2&&(r==-1?D=e.cn(A,u):D=d.Math.min(31,r+((t.u[n]&Q1)>>18!=1&&(t.u[n]&Q1)>>18!=2?-1:(t.u[n]&F0)>>20))),A.C[i]=t.C[n],A.s[i]=t.s[n],A.A[i]=t.A[n],A.u[i]=t.u[n],A.B[i]=A.K?t.B[n]:0,e.ji(A.J[i],t.J[n]),A.w[i]=t.w[n],A.v!=null&&(A.v[i]=null),t.v!=null&&t.v[n]!=null&&A.K&&(A.v==null&&(A.v=j(l.j7,E1,7,A.C.length,0,2)),A.v[i]=y.Jqb(t.v[n],t.v[n].length)),A.t!=null&&(A.t[i]=null),t.t!=null&&t.t[n]!=null&&(A.t==null&&(A.t=j(l.f7,bA,10,A.C.length,0,2)),A.t[i]=y.Hqb(t.t[n],t.t[n].length)),D!=-1&&(A.u[i]&=-32505857,A.u[i]|=D<<20),++A.q,A.T=0,i;},e.ml=function(t,A,n,P,r,i,D,u){var o,Q,L,s;return Q=A.r,Q>=A.N&&e.Sn(A,A.N*2),s=(t.F[n]&768)>>8,L=-1,s==1&&(P==-1?L=e.cn(A,s):L=d.Math.min(32,P+((t.F[n]&768)>>8!=1&&(t.F[n]&768)>>8!=2?-1:(t.F[n]&E0)>>10))),s==2&&(r==-1?L=e.cn(A,s):L=d.Math.min(32,r+((t.F[n]&768)>>8!=1&&(t.F[n]&768)>>8!=2?-1:(t.F[n]&E0)>>10))),A.D[0][Q]=i,A.D[1][Q]=D,o=u&&t.Zb(n)?8:t.H[n],A.H[Q]=o,A.F[Q]=t.F[n],A.G[Q]=A.K?t.G[n]:0,L!=-1&&(A.F[Q]&=-31745,A.F[Q]|=L<<10),++A.r,A.T=0,Q;},e.nl=function(t,A,n,P,r,i,D){return e.ml(t,A,n,P,r,i==null?t.D[0][n]:i[t.D[0][n]],i==null?t.D[1][n]:i[t.D[1][n]],D);},e.ol=function(t,A){var n,P;for(A.v=null,A.t=null,A.K=t.K,A.q=0,n=0;n<t.q;n++)e.ll(t,A,n,0,0);for(A.r=0,P=0;P<t.r;P++)e.ml(t,A,P,0,0,t.D[0][P],t.D[1][P],false);t.Yb(A);},e.pl=function(t,A){A.K=t.K,A.L=t.L,A.S=t.S,A.I=t.I,A.P=t.P,A.T=t.T&24;},e.ql=function(t,A){var n,P;for(n=0;n<t.r;n++)for(P=0;P<2;P++)t.D[P][n]==A&&(t.H[n]=512);e.an(t,t.w[A]),t.C[A]=-1,t.v!=null&&(t.v[A]=null),t.t!=null&&(t.t[A]=null),e.kl(t),t.T=0;},e.rl=function(t,A,n){var P,r;return P=e.xl(t,A,n),P!=-1?((t.u[P]&512)!=0?e.wl(t):e.ql(t,P),t.T=0,true):(r=e.yl(t,A,n),r!=-1?((t.u[t.D[0][r]]&t.u[t.D[1][r]]&512)!=0?e.wl(t):e.ul(t,r),t.T=0,true):false);},e.sl=function(t,A){var n,P,r,i;if(A.length==0)return null;for(P=A,r=0,i=P.length;r<i;++r)n=P[r],t.C[n]=-1;return e.vl(t);},e.tl=function(t,A){t.H[A]=512,e.kl(t),t.T=0;},e.ul=function(t,A){var n,P,r;for(P=0;P<2;P++){for(n=0,r=0;r<t.r;r++)r!=A&&(t.D[0][r]==t.D[P][A]||t.D[1][r]==t.D[P][A])&&++n;n==0&&(e.an(t,t.w[t.D[P][A]]),t.C[t.D[P][A]]=-1);}t.H[A]=512,e.kl(t),t.T=0;},e.vl=function(t){var A,n,P;for(P=false,A=0;A<t.q;A++)t.C[A]==-1&&(P=true,e.an(t,t.w[A]));for(n=0;n<t.r;n++)t.H[n]==512?P=true:(t.C[t.D[0][n]]==-1||t.C[t.D[1][n]]==-1)&&(t.H[n]=512,P=true);return P?(t.T=0,e.kl(t)):null;},e.wl=function(t){var A,n;for(n=false,A=0;A<t.q;A++)(t.u[A]&512)!=0&&(t.C[A]=-1,n=true);return n&&e.vl(t)!=null;},e.xl=function(t,A,n){var P,r,i,D,u,o,Q,L;for(D=-1,r=e.Wl(t,t.q,t.r,e.Qk),u=U0,o=r*r/12,P=0;P<t.q;P++)Q=t.J[P].a,L=t.J[P].b,i=(A-Q)*(A-Q)+(n-L)*(n-L),i<o&&i<u&&(u=i,D=P);return D;},e.yl=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;for(h=-1,c=e.Wl(t,t.q,t.r,e.Qk),b=U0,P=0;P<t.r;P++)g=t.J[t.D[0][P]].a,k=t.J[t.D[0][P]].b,B=t.J[t.D[1][P]].a,M=t.J[t.D[1][P]].b,L=B-g,s=M-k,r=d.Math.sqrt(L*L+s*s),i=(g+B)/2,D=(k+M)/2,L=A-i,s=n-D,!(d.Math.sqrt(L*L+s*s)>r/2)&&(B==g?Q=d.Math.abs(g-A):(u=(M-k)/(g-B),o=-u*g-k,Q=d.Math.abs((u*A+n+o)/d.Math.sqrt(u*u+1))),Q<c&&Q<b&&(b=Q,h=P));return h;},e.zl=function(t,A){return ((t.u[A]&cA)>>>27)-1;},e.Al=function(t,A){return (t.u[A]&49152)>>14;},e.Bl=function(t,A){return t.s[A];},e.Cl=function(t,A){return t.u[A]&448;},e.Dl=function(t,A){return t.J[A];},e.El=function(t,A){var n;return t.t==null||t.t[A]==null?null:T.ijb((n=t.t[A],S1.Mmb(),n));},e.Fl=function(t,A){return t.t==null?null:t.t[A];},e.Gl=function(t,A){return (t.u[A]&Q1)>>18!=1&&(t.u[A]&Q1)>>18!=2?-1:(t.u[A]&F0)>>20;},e.Hl=function(t,A){return (t.u[A]&Q1)>>18;},e.Il=function(t,A){return e.Lk[t.C[A]];},e.Jl=function(t,A){return t.v==null?null:t.v[A];},e.Kl=function(t,A){var n,P,r;if(t.v==null||t.v[A]==null)return pt(J(t.B[A],1),0)?"":e.Lk[t.C[A]];for(r="",P=0;P<t.v[A].length;P++)P>0&&(r=(w.Qzb(r),r+(w.Qzb(","),","))),n=t.v[A][P],r=T.gjb(r,e.Lk[n]);return r;},e.Ll=function(t,A){return d.Math.abs(t.w[A]);},e.Ml=function(t,A){return t.A[A];},e.Nl=function(t,A){return t.u[A]&3;},e.Ol=function(t,A){return t.B[A];},e.Pl=function(t,A){return t.u[A]&48;},e.Ql=function(t,A){return t.J[A].a;},e.Rl=function(t,A){return t.J[A].b;},e.Sl=function(t,A){return t.J[A].c;},e.Tl=function(t,A){return t.C[A];},e.Ul=function(t){return e.Wl(t,t.q,t.r,e.Qk);},e.Vl=function(t,A){return e.Wl(t,t.q,t.r,A);},e.Wl=function(t,A,n,P){return e.Xl(t,A,n,P,t.J);},e.Xl=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c;for(s=false,h=0,Q=0;Q<n;Q++)t.H[Q]!=16&&(t.G[Q]&UA)==0&&++h;if(h==0){for(L=0;L<n;L++)(t.G[L]&UA)==0&&++h;s=true;}if(h==0){if(A<2)return P;for(c=U0,i=1;i<A;i++)for(D=0;D<i;D++)b=e.di(r[i],r[D]),b>0&&b<c&&(c=b);return c!=U0?.6*c:P;}for(u=0,o=0;o<n;o++)(s||t.H[o]!=16)&&(t.G[o]&UA)==0&&(u+=e.di(r[t.D[1][o]],r[t.D[0][o]]));return u/h;},e.Yl=function(t,A,n){return e.no(t.J[A].a,t.J[A].b,t.J[n].a,t.J[n].b);},e.Zl=function(t,A,n){return t.D[A][n];},e.$l=function(t,A){return ((t.G[A]&7680)>>9)+((t.G[A]&$n)>>13);},e._l=function(t,A){return (t.G[A]&7680)>>9;},e.am=function(t,A){return (t.F[A]&48)>>4;},e.bm=function(t,A){return (t.F[A]&768)>>8!=1&&(t.F[A]&768)>>8!=2?-1:(t.F[A]&E0)>>10;},e.cm=function(t,A){return (t.F[A]&768)>>8;},e.dm=function(t,A){var n,P,r,i;return n=t.D[0][A],P=t.D[1][A],r=t.J[P].a-t.J[n].a,i=t.J[P].b-t.J[n].b,d.Math.sqrt(r*r+i*i);},e.em=function(t,A,n){var P;for(P=0;P<t.r;P++)if((t.D[0][P]==A&&t.D[1][P]==n||t.D[0][P]==n&&t.D[1][P]==A)&&t.H[P]!=512)return P;return  -1;},e.fm=function(t,A){if(t.K&&(t.G[A]&127)!=0){if((t.G[A]&9)!=0)return 1;if((t.G[A]&2)!=0)return 2;if((t.G[A]&4)!=0)return 3;if((t.G[A]&32)!=0)return 4;if((t.G[A]&64)!=0)return 5;if((t.G[A]&16)!=0)return 0;}switch(t.H[A]&127){case 1:case 8:return 1;case 2:return 2;case 4:return 3;case 32:return 4;case 64:return 5;default:return 0;}},e.gm=function(t,A){return t.F[A]&3;},e.hm=function(t,A){return t.G[A];},e.im=function(t,A){return t.H[A];},e.jm=function(t,A){return t.H[A]&127;},e.km=function(t,A){var n,P,r,i,D;if(t.q==0)return null;for(P=t.J[0].a,i=t.J[0].b,r=t.J[0].a,D=t.J[0].b,n=1;n<t.q;n++)P>t.J[n].a?P=t.J[n].a:r<t.J[n].a&&(r=t.J[n].a),i>t.J[n].b?i=t.J[n].b:D<t.J[n].b&&(D=t.J[n].b);return A?(A.c=P,A.d=i,A.b=r-P,A.a=D-i):A=new Tt.KN(P,i,r-P,D-i),A;},e.lm=function(t,A){var n;return n=t.C[A]<e.Mk.length?e.Mk[t.C[A]]:null,n==null?6:n[n.length-1];},e.mm=function(t,A,n){return e.nm(t,A,n,t.C[A]);},e.nm=function(t,A,n,P){var r,i;return P>=171&&P<=190?0:(i=0,(t.u[A]&48)==32&&(i-=1),((t.u[A]&48)==16||(t.u[A]&48)==48)&&(i-=2),r=t.s[A],r==0&&t.K&&(H1(J(t.B[A],mA),oe)&&(r=-1),H1(J(t.B[A],mA),ue)&&(r=1)),P==7||P==8||P==9?i+=r:P==6||P==14||P==32?i-=d.Math.abs(r):P==15||P==33?n-i-r<=3?i+=r:i-=r:P==16||P==34||P==52?n-i-r<=4?i+=r:i-=d.Math.abs(r):P==17||P==35||P==53?n-i-r<=5?i+=r:i-=d.Math.abs(r):i-=r,i);},e.om=function(t,A){var n;return n=e.pm(t,A),n+e.nm(t,A,n,t.C[A]);},e.pm=function(t,A){var n,P;return n=((t.u[A]&cA)>>>27)-1,n==-1&&(n=(P=t.C[A]<e.Mk.length?e.Mk[t.C[A]]:null,P==null?6:P[P.length-1])),n;},e.qm=function(t,A){var n,P,r,i,D;for(i=e.Tm(t,t.D[0][A])||e.Tm(t,t.D[1][A])?5:3,P=0;P<2;P++)n=t.D[P][A],r=e.fm(t,A)+(D=e.pm(t,n),D+e.nm(t,n,D,t.C[n]))-e.vp(t,n),i>r&&(i=r);return i;},e.rm=function(t,A){return (t.u[A]&q1)!=0;},e.sm=function(t,A){var n,P,r;return P=e.qm(t,A),n=e.Lm(t,t.D[0][A])||e.Lm(t,t.D[1][A]),r=n?16:1,t.H[A]==64?(t.H[A]=r,t.T=0,true):t.H[A]==32?(t.H[A]=P>4?64:r,t.T=0,true):t.H[A]==4?(t.H[A]=P>3?32:r,t.T=0,true):t.H[A]==2&&(t.H[A]=386,t.T&=7,(t.F[A]&128)==0)?true:t.H[A]==386?(P>2?t.H[A]=4:t.H[A]=r,t.T=0,true):(384&t.H[A])!=0?(t.H[A]=1,t.T&=7,true):!n&&P<2?false:t.H[A]==1?(t.H[A]=2,t.T=0,true):P<1?false:t.H[A]==16?(t.H[A]=1,t.T=0,true):false;},e.tm=function(t){var A;for(t.T=0,t.C=j(l.j7,H,6,t.M,15,1),t.s=j(l.j7,H,6,t.M,15,1),t.w=j(l.j7,H,6,t.M,15,1),t.J=j(l.C7,qe,29,t.M,0,1),A=0;A<t.M;A++)t.J[A]=new e.ni();t.A=j(l.j7,H,6,t.M,15,1),t.u=j(l.j7,H,6,t.M,15,1),t.B=j(l.k7,e0,6,t.M,14,1),t.v=null,t.t=null,t.D=y1(l.j7,[E1,H],[7,6],15,[2,t.N],2),t.H=j(l.j7,H,6,t.N,15,1),t.F=j(l.j7,H,6,t.N,15,1),t.G=j(l.j7,H,6,t.N,15,1);},e.um=function(t,A){t.T&=~A;},e.vm=function(t){var A;for(A=0;A<t.q;A++)if(t.J[A].c!=0)return  true;return  false;},e.wm=function(t,A){return (t.u[A]&ln)!=0;},e.xm=function(t,A){return t.C[A]==-1;},e.ym=function(t,A){return (t.u[A]&4)!=0;},e.zm=function(t,A){return (t.u[A]&S0)!=0;},e.Am=function(t,A){return t.w[A]<0;},e.Bm=function(t,A){return (t.F[A]&LA)!=0;},e.Cm=function(t,A){return (t.G[A]&UA)!=0;},e.Dm=function(t,A){return (t.F[A]&q1)!=0;},e.Em=function(t,A){return t.H[A]==512;},e.Fm=function(t,A){return (t.F[A]&4)!=0;},e.Gm=function(t,A){return (t.F[A]&n0)!=0;},e.Hm=function(t,A){var n,P,r,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(P=t.v[A],r=0,i=P.length;r<i;++r)if(n=P[r],!e.qo(n))return  false;}}return e.qo(t.C[A]);},e.Im=function(t,A){var n,P,r,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(P=t.v[A],r=0,i=P.length;r<i;++r)if(n=P[r],!e.ro(n))return  false;}}return e.ro(t.C[A]);},e.Jm=function(t,A){return pt(J(t.B[A],r0),0);},e.Km=function(t,A){return (t.u[A]&n0)!=0;},e.Lm=function(t,A){var n,P,r,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(P=t.v[A],r=0,i=P.length;r<i;++r)if(n=P[r],!(n>=3&&n<=4||n>=11&&n<=13||n>=19&&n<=31||n>=37&&n<=51||n>=55&&n<=84||n>=87&&n<=112))return  false;}}return e.so(t.C[A]);},e.Mm=function(t,A){return t.A[A]==0;},e.Nm=function(t,A){var n,P,r,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(P=t.v[A],r=0,i=P.length;r<i;++r)if(n=P[r],!(n==1||n>=5&&n<=9||n>=14&&n<=17||n>=32&&n<=35||n>=52&&n<=53))return  false;}}return e.uo(t.C[A]);},e.Om=function(t){var A;for(A=0;A<t.q;A++)switch(t.C[A]){case 1:case 5:case 6:case 7:case 8:case 9:case 14:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:continue;default:return  false;}return  true;},e.Pm=function(t,A){return (t.u[A]&512)!=0;},e.Qm=function(t,A){return (t.u[t.D[0][A]]&t.u[t.D[1][A]]&512)!=0;},e.Rm=function(t,A){return t.H[A]==257||t.H[A]==129;},e.Sm=function(t,A,n){return (t.H[A]==257||t.H[A]==129)&&t.D[0][A]==n;},e.Tm=function(t,A){var n,P,r,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(P=t.v[A],r=0,i=P.length;r<i;++r)if(n=P[r],!(n>=21&&n<=30||n>=39&&n<=48||n==57||n>=72&&n<=80||n==89||n>=104&&n<=112))return  false;}}return e.vo(t.C[A]);},e.Um=function(t,A){t.C[A]=-1;},e.Vm=function(t,A){t.H[A]=512;},e.Wm=function(t,A,n,P,r,i,D,u){var o,Q,L,s,h,b,c,g,B,k;for(A!=P&&(B=t.J[A].a-t.J[P].a,k=t.J[A].b-t.J[P].b,u=d.Math.sqrt(B*B+k*k)),o=A,L=e.Wn(t,A)!=3,g=1;g<n;g++){for(h=t.J[o].a+u*d.Math.sin(i),b=t.J[o].b+u*d.Math.cos(i),c=-1,s=0;s<t.q;s++)if(d.Math.abs(h-t.J[s].a)<4&&d.Math.abs(b-t.J[s].b)<4){c=s;break;}c==-1&&(c=e.Tk(t,h,b,0),t.J[c].a=h,t.J[c].b=b,t.J[c].c=0),Q=e.em(t,o,c),Q==-1&&(Q=e.Wk(t,o,c,e.Lm(t,o)||e.Lm(t,c)?16:1),r&&(L&&e.Wn(t,t.D[0][Q])<4&&e.Wn(t,t.D[1][Q])<3&&(t.H[Q]=2),L=!L)),o=c,i+=D;}Q=e.em(t,o,P),Q==-1&&(Q=e.Wk(t,o,P,e.Lm(t,o)||e.Lm(t,P)?16:1)),r&&L&&e.Wn(t,t.D[0][Q])<4&&e.Wn(t,t.D[1][Q])<4&&(t.H[Q]=2);},e.Xm=function(t,A,n){var P,r;if(e.fm(t,A)!=1)return  false;if((t.u[n]&3)!=0)return  true;for(r=0;r<t.r;r++)if(r!=A&&t.H[r]==2&&(t.D[0][r]==n&&(t.u[t.D[1][r]]&3)!=0||t.D[1][r]==n&&(t.u[t.D[0][r]]&3)!=0))return  true;for(P=0;P<t.r;P++)if(P!=A&&t.H[P]==1&&(t.D[0][P]==n||t.D[1][P]==n)&&(t.F[P]&3)!=0)return  true;return  false;},e.Ym=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=-449;},e.Zm=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=an;},e.$m=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=-513;},e._m=function(t){var A;for(A=0;A<t.r;A++)t.F[A]&=-98305;},e.an=function(t,A){var n;for(n=0;n<t.q;n++)d.Math.abs(t.w[n])==d.Math.abs(A)&&(t.w[n]=0);},e.bn=function(t){var A,n,P,r;for(r=false,n=0;n<t.q;n++)pt(J(t.B[n],r0),0)&&(t.C[n]=-1,r=true);for(r&&e.vl(t),t.v!=null&&(t.v=null,r=true),A=0;A<t.q;A++)pt(t.B[A],0)&&(t.B[A]=0,r=true);for(P=0;P<t.r;P++)t.G[P]!=0&&(t.G[P]=0,r=true),t.H[P]==8&&(t.H[P]=1,r=true);return r&&(t.T=0),r;},e.cn=function(t,A){var n,P,r,i,D,u,o,Q,L;if(A==0)return 0;for(u=null,P=0;P<t.q;P++)(t.u[P]&Q1)>>18==A&&(u==null&&(u=j(l.Feb,ht,6,32,16,1)),u[(t.u[P]&Q1)>>18!=1&&(t.u[P]&Q1)>>18!=2?-1:(t.u[P]&F0)>>20]=true);for(i=0;i<t.r;i++)(t.F[i]&768)>>8==A&&(u==null&&(u=j(l.Feb,ht,6,32,16,1)),u[(t.F[i]&768)>>8!=1&&(t.F[i]&768)>>8!=2?-1:(t.F[i]&E0)>>10]=true);if(L=0,u!=null){for(Q=j(l.j7,H,6,32,15,1),o=0;o<32;o++)u[o]&&(Q[o]=L++);for(n=0;n<t.q;n++)(t.u[n]&Q1)>>18==A&&(D=Q[(t.u[n]&Q1)>>18!=1&&(t.u[n]&Q1)>>18!=2?-1:(t.u[n]&F0)>>20],t.u[n]&=-32505857,t.u[n]|=D<<20);for(r=0;r<t.r;r++)(t.F[r]&768)>>8==A&&(D=Q[(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10],t.F[r]&=-31745,t.F[r]|=D<<10);}return L;},e.dn=function(t,A){var n;for(n=0;n<t.q;n++)t.J[n].a*=A,t.J[n].b*=A;},e.en=function(t,A){t.q=A,t.T=0;},e.fn=function(t,A){t.r=A,t.T=0;},e.gn=function(t,A,n){n>=-1&&n<=14&&(t.u[A]&=-2013265921,t.u[A]|=1+n<<27,t.C[A]==6&&(n==-1||n==0||n==2||n==4)&&(t.u[A]&=-49,n==2&&(t.u[A]|=16)));},e.hn=function(t,A,n){t.u[A]&=-49153,t.u[A]|=n<<14;},e.jn=function(t,A,n){t.s[A]=n,t.T=0;},e.kn=function(t,A,n){t.u[A]&=-449,t.u[A]|=n;},e.ln=function(t,A,n){n?t.u[A]|=ln:t.u[A]&=tP,t.T&=7;},e.mn=function(t,A,n){var P,r,i;if(n!=null&&(w.aAb(n).length==0?n=null:(P=e.po(n,321),(P!=0&&P==t.C[A]||T.kjb(n,"?"))&&(e.Cn(t,A,P),n=null))),n==null){if(t.t!=null&&t.t[A]!=null)return t.t[A]=null,true;}else if(t.t==null&&(t.t=j(l.f7,bA,10,t.M,0,2)),t.t[A]==null||!T.kjb(T.ijb((i=t.t[A],S1.Mmb(),i)),n))return t.t[A]=w.Czb((r=n,S1.Mmb(),r)),true;return  false;},e.nn=function(t,A,n){n!=null&&n.length==0&&(n=null),n==null?t.t!=null&&(t.t[A]=null):(t.t==null&&(t.t=j(l.f7,bA,10,t.M,0,2)),t.t[A]=n);},e.on=function(t,A,n,P){var r,i,D;if(n==0)t.u[A]&=AP,t.u[A]|=n<<18;else {if(P>=32)return;if(P==-1){for(D=-1,i=0;i<t.q;i++)i!=A&&n==(t.u[i]&Q1)>>18&&D<((t.u[i]&Q1)>>18!=1&&(t.u[i]&Q1)>>18!=2?-1:(t.u[i]&F0)>>20)&&(D=(t.u[i]&Q1)>>18!=1&&(t.u[i]&Q1)>>18!=2?-1:(t.u[i]&F0)>>20);for(r=0;r<t.r;r++)n==(t.F[r]&768)>>8&&D<((t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10)&&(D=(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10);if(P=D+1,P>=32)return;}t.u[A]&=AP,t.u[A]|=n<<18|P<<20;}t.T&=7;},e.pn=function(t,A,n){t.v==null&&(t.v=j(l.j7,E1,7,t.M,0,2)),w.vzb(n,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),t.v[A]=n,t.T=0,t.K=true;},e.qn=function(t,A,n,P){var r;if(n==null){t.v!=null&&(t.v[A]=null);return;}if(n.length==1&&!P){r=n[0],t.C[A]!=r&&e.fl(t,A,r,0,-1,0),t.v!=null&&(t.v[A]=null);return;}t.v==null&&(t.v=j(l.j7,E1,7,t.M,0,2)),t.v[A]=n,P&&(t.B[A]=rt(t.B[A],1)),t.T=0,t.K=true;},e.rn=function(t,A,n,P){t.w[A]=P?-n:n;},e.sn=function(t,A,n){n?t.u[A]|=n0:t.u[A]&=an;},e.tn=function(t,A,n){t.A[A]=n,t.T&=t.C[A]==1?0:7;},e.un=function(t,A,n,P){t.u[A]&=-8,n!=3&&(t.u[A]&=tP),t.u[A]|=n,P&&(t.u[A]|=4);},e.vn=function(t,A,n,P){P?t.B[A]=rt(t.B[A],n):t.B[A]=J(t.B[A],tA(n)),t.T=0,t.K=true;},e.wn=function(t,A,n){t.u[A]&=-49,t.u[A]|=n,t.T&=7;},e.xn=function(t,A,n){n?t.u[A]|=512:t.u[A]&=-513;},e.yn=function(t,A,n){t.u[A]&=-67108865,n&&(t.u[A]|=S0);},e.zn=function(t,A,n){t.J[A].a=n,t.T&=7;},e.An=function(t,A,n){t.J[A].b=n,t.T&=7;},e.Bn=function(t,A,n){t.J[A].c=n,t.T&=7;},e.Cn=function(t,A,n){n>=0&&n<=190&&(n==151||n==152?(t.C[A]=1,t.A[A]=n-149):(t.C[A]=n,t.A[A]=0),t.u[A]&=-2013265921,t.T=0);},e.Dn=function(t,A,n,P){t.D[A][n]=P,t.T=0;},e.En=function(t,A,n){n?t.F[A]|=LA:t.F[A]&=-32769;},e.Fn=function(t,A,n){t.F[A]&=-49,t.F[A]|=n<<4;},e.Gn=function(t,A,n,P){var r,i,D;if(n==0)t.F[A]&=-32513,t.F[A]|=n<<8;else {if(P>=32)return;if(P==-1){for(D=-1,i=0;i<t.q;i++)n==(t.u[i]&Q1)>>18&&D<((t.u[i]&Q1)>>18!=1&&(t.u[i]&Q1)>>18!=2?-1:(t.u[i]&F0)>>20)&&(D=(t.u[i]&Q1)>>18!=1&&(t.u[i]&Q1)>>18!=2?-1:(t.u[i]&F0)>>20);for(r=0;r<t.r;r++)r!=A&&n==(t.F[r]&768)>>8&&D<((t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10)&&(D=(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10);if(P=D+1,P>=32)return;}t.F[A]&=-32513,t.F[A]|=n<<8|P<<10;}t.T&=7;},e.Hn=function(t,A,n){n?t.F[A]|=q1:t.F[A]&=-65537;},e.In=function(t,A,n){t.H[A]=n==1?1:n==2?2:n==3?4:16,t.T=0;},e.Jn=function(t,A,n,P){t.F[A]&=-131080,t.F[A]|=n,P&&(t.F[A]|=4);},e.Kn=function(t,A){t.F[A]|=n0;},e.Ln=function(t,A,n,P){P?t.G[A]|=n:t.G[A]&=~n,t.T=0,P&&n!=0&&(t.K=true);},e.Mn=function(t,A,n){t.H[A]=n,t.T=0;},e.Nn=function(t,A){t.I=A;},e.On=function(t,A){t.K!=A&&(t.K=A,A||e.bn(t),t.T=0);},e.Pn=function(t,A){t.T=A;},e.Qn=function(t,A){t.S=A;},e.Rn=function(t,A){var n,P;for(t.C=e.ho(t.C,A),t.s=e.ho(t.s,A),t.w=e.ho(t.w,A),P=t.J.length,t.J=e.jo(t.J,A),n=P;n<A;n++)t.J[n]=new e.ni();t.A=e.ho(t.A,A),t.u=e.ho(t.u,A),t.B=e.io(t.B,A),t.v!=null&&(t.v=e.lo(t.v,A)),t.t!=null&&(t.t=e.ko(t.t,A)),t.M=A;},e.Sn=function(t,A){t.D[0]=e.ho(t.D[0],A),t.D[1]=e.ho(t.D[1],A),t.H=e.ho(t.H,A),t.F=e.ho(t.F,A),t.G=e.ho(t.G,A),t.N=A;},e.Tn=function(t,A){t.O=A;},e.Un=function(t,A){t.P=A;},e.Vn=function(t,A){t.u[A]|=q1;},e.Wn=function(t,A){var n,P;for(P=0,n=0;n<t.r;n++)(t.D[0][n]==A||t.D[1][n]==A)&&(P+=e.fm(t,n));return P;},e.Xn=function(t){var A,n,P;for(n=false,P=false,A=0;A<t.q;A++)t.A[A]!=0&&(t.A[A]=0,n=true,t.C[A]==1&&(P=true));return P&&(t.T=0),n;},e.Yn=function(t,A,n){return e.Lm(t,A)||e.Lm(t,n)?16:1;},e.Zn=function(t,A,n){var P,r,i,D,u,o;for(D=t.C[A],t.C[A]=t.C[n],t.C[n]=D,D=t.s[A],t.s[A]=t.s[n],t.s[n]=D,D=t.A[A],t.A[A]=t.A[n],t.A[n]=D,D=t.u[A],t.u[A]=t.u[n],t.u[n]=D,o=t.B[A],t.B[A]=t.B[n],t.B[n]=o,D=t.w[A],t.w[A]=t.w[n],t.w[n]=D,i=t.J[A],t.J[A]=t.J[n],t.J[n]=i,t.v!=null&&(u=t.v[A],t.v[A]=t.v[n],t.v[n]=u),t.t!=null&&(u=t.t[A],t.t[A]=t.t[n],t.t[n]=u),P=0;P<t.r;P++)for(r=0;r<2;r++)t.D[r][P]==A?t.D[r][P]=n:t.D[r][P]==n&&(t.D[r][P]=A);t.T=0;},e.$n=function(t,A,n){var P;P=t.D[0][A],t.D[0][A]=t.D[0][n],t.D[0][n]=P,P=t.D[1][A],t.D[1][A]=t.D[1][n],t.D[1][n]=P,P=t.H[A],t.H[A]=t.H[n],t.H[n]=P,P=t.F[A],t.F[A]=t.F[n],t.F[n]=P,P=t.G[A],t.G[A]=t.G[n],t.G[n]=P,t.T=0;},e._n=function(t,A,n){var P;for(P=0;P<t.q;P++)t.J[P].a+=A,t.J[P].b+=n;t.U+=A,t.V+=n;},e.ao=function(t,A,n){var P,r;switch(r=n&127,P=e.qm(t,A),r){case 1:case 8:return P>=1;case 2:return P>=2;case 4:return P>=3;case 32:return P>=4;case 64:return P>=5;case 16:return  true;default:return  false;}},e.bo=function(t,A,n,P){var r,i,D;for(r=0;r<t.q;r++)(!P||(t.u[r]&512)!=0)&&(D=t.R[r]*A,i=t.Q[r]-n,t.J[r].a=t.U+D*d.Math.sin(i),t.J[r].b=t.V+D*d.Math.cos(i));P&&(t.T&=7);},e.co=function(t,A,n){var P,r,i;for(t.U=A,t.V=n,t.Q=j(l.h7,Ft,6,t.q,15,1),t.R=j(l.h7,Ft,6,t.q,15,1),P=0;P<t.q;P++)r=A-t.J[P].a,i=n-t.J[P].b,t.R[P]=d.Math.sqrt(r*r+i*i),t.Q[P]=e.no(A,n,t.J[P].a,t.J[P].b);},e.eo=function(){e.Rk(),this.M=this.N=256,e.tm(this);},e.fo=function(t,A){this.M=d.Math.max(1,t),this.N=d.Math.max(1,A),e.tm(this);},e.ho=function(t,A){var n;return n=j(l.j7,H,6,A,15,1),T.$jb(t,0,n,0,d.Math.min(t.length,A)),n;},e.io=function(t,A){var n;return n=j(l.k7,e0,6,A,14,1),T.$jb(t,0,n,0,d.Math.min(t.length,A)),n;},e.jo=function(t,A){var n,P;for(n=j(l.C7,qe,29,A,0,1),P=0;P<t.length;P++)t[P]&&(n[P]=new e.pi(t[P]));return n;},e.ko=function(t,A){var n,P;for(n=j(l.f7,bA,10,A,0,2),P=0;P<t.length;P++)t[P]!=null&&(n[P]=j(l.f7,Y,6,t[P].length,15,1),T.$jb(t[P],0,n[P],0,t[P].length));return n;},e.lo=function(t,A){var n,P;for(n=j(l.j7,E1,7,A,0,2),P=0;P<t.length;P++)t[P]!=null&&(n[P]=j(l.j7,H,6,t[P].length,15,1),T.$jb(t[P],0,n[P],0,t[P].length));return n;},e.mo=function(t){return e.Rk(),t>=0&&t<e.Mk.length&&e.Mk[t]!=null?e.Mk[t]:t>=171&&t<=190?e.Kk:e.Ok;},e.no=function(t,A,n,P){e.Rk();var r,i,D;return i=n-t,D=P-A,D!=0?(r=d.Math.atan(i/D),D<0&&(i<0?r-=A1:r+=A1)):r=i>0?k0:un,r;},e.oo=function(t,A){e.Rk();var n;for(n=t-A;n<lr;)n+=I1;for(;n>A1;)n-=I1;return n;},e.po=function(t,A){e.Rk();var n,P,r,i,D;if((A&256)!=0&&T.kjb(t,"?"))return 0;for(P=1;P<=128;P++)if(!T.kjb(t,R1)&&T.ljb(t,e.Lk[P]))return P;if((A&2)!=0){for(r=129;r<=144;r++)if(T.ljb(t,e.Lk[r]))return r;}if((A&4)!=0){for(i=146;i<=148;i++)if(T.ljb(t,e.Lk[i]))return i;}if((A&1)!=0){for(D=151;D<=152;D++)if(T.ljb(t,e.Lk[D]))return D;}if((A&32)!=0&&T.ljb(t,e.Lk[153]))return 153;if((A&8)!=0&&T.ljb(t,e.Lk[154]))return 154;if((A&16)!=0&&T.ljb(t,e.Lk[145]))return 145;if((A&128)!=0&&T.ljb(t,e.Lk[159]))return 159;if((A&64)!=0){for(n=171;n<=190;n++)if(T.ljb(t,e.Lk[n]))return n;}return 0;},e.qo=function(t){switch(e.Rk(),t){case 7:case 8:case 9:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:return  true;}return  false;},e.ro=function(t){return e.Rk(),!(t==1||t==6||e.qo(t)||t==2||t==10||t==18||t==36||t==54||t==86||t>103);},e.so=function(t){return t>=3&&t<=4||t>=11&&t<=13||t>=19&&t<=31||t>=37&&t<=51||t>=55&&t<=84||t>=87&&t<=112;},e.to=function(t){return t==2||t==10||t==18||t==36||t==54||t==86;},e.uo=function(t){return t==1||t>=5&&t<=9||t>=14&&t<=17||t>=32&&t<=35||t>=52&&t<=53;},e.vo=function(t){return t>=21&&t<=30||t>=39&&t<=48||t==57||t>=72&&t<=80||t==89||t>=104&&t<=112;},N(88,1,{88:1,4:1},e.eo),U.Yb=function(t){e.pl(this,t);},U.Zb=function(t){return this.H[t]==8;},U.q=0,U.r=0,U.I=0,U.K=false,U.L=false,U.M=0,U.N=0,U.O=0,U.S=false,U.T=0,U.U=0,U.V=0,e.Qk=24,l.R7=O(88),e.xo=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h;for(e.bw(A,1),P==null&&(P=j(l.j7,H,6,A.q,15,1)),u=e.cn(t,1),o=e.cn(t,2),h=j(l.Feb,ht,6,A.q,16,1),Q=j(l.j7,H,6,A.q,15,1),Q[0]=n,h[n]=true,P[n]=e.ll(A,t,n,u,o),D=0,L=0;D<=L;){for(s=0;s<e.Lo(A,Q[D]);s++)i=A.i[Q[D]][s],h[i]||(Q[++L]=i,h[i]=true,P[i]=e.ll(A,t,i,u,o));++D;}for(r=0;r<A.r;r++)h[A.D[0][r]]&&e.ml(A,t,r,u,o,P==null?A.D[0][r]:P[A.D[0][r]],P==null?A.D[1][r]:P[A.D[1][r]],false);e.cn(t,1),e.cn(t,2),t.T=0;},e.yo=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h;for(t.j=j(l.j7,H,6,t.q,15,1),t.e=j(l.j7,H,6,t.q,15,1),t.i=j(l.j7,E1,7,t.q,0,2),t.k=j(l.j7,E1,7,t.q,0,2),t.n=j(l.j7,E1,7,t.q,0,2),t.o=j(l.j7,H,6,t.f,15,1),Q=j(l.j7,H,6,t.q,15,1),D=0;D<t.r;D++)++Q[t.D[0][D]],++Q[t.D[1][D]];for(P=0;P<t.q;P++)t.i[P]=j(l.j7,H,6,Q[P],15,1),t.k[P]=j(l.j7,H,6,Q[P],15,1),t.n[P]=j(l.j7,H,6,Q[P],15,1);for(s=false,u=0;u<t.g;u++){if(h=e.fm(t,u),h==0){s=true;continue;}for(L=0;L<2;L++)n=t.D[L][u],A=t.e[n],t.n[n][A]=h,t.i[n][A]=t.D[1-L][u],t.k[n][A]=u,++t.e[n],++t.j[n],n<t.f&&(h>1?t.o[n]+=h-1:t.H[u]==8&&(t.o[n]=1));}for(o=t.g;o<t.r;o++){if(h=e.fm(t,o),h==0){s=true;continue;}for(L=0;L<2;L++)n=t.D[L][o],A=t.e[n],t.n[n][A]=h,t.i[n][A]=t.D[1-L][o],t.k[n][A]=o,++t.e[n],t.D[1-L][o]<t.f&&++t.j[n];}if(s){for(A=j(l.j7,H,6,t.q,15,1),r=0;r<t.q;r++)A[r]=t.e[r];for(i=0;i<t.r;i++)if(h=e.fm(t,i),h==0)for(L=0;L<2;L++)n=t.D[L][i],t.n[n][A[n]]=h,t.i[n][A[n]]=t.D[1-L][i],t.k[n][A[n]]=i,++A[n];}},e.zo=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V;for(e.bw(t,1),n&&(A=true),o=0;o<t.r;o++)if(Q=e.fm(t,o),Q==1||Q==2){if(t.s[t.D[0][o]]>0&&t.s[t.D[1][o]]<0)i=t.D[0][o],D=t.D[1][o];else if(t.s[t.D[0][o]]<0&&t.s[t.D[1][o]]>0)i=t.D[1][o],D=t.D[0][o];else continue;if(e.Lm(t,i)||e.Lm(t,D)||t.C[i]<9&&e.vp(t,i)>3||t.C[D]<9&&e.vp(t,D)>3)continue;s=e.op(t,i)!=0,t.s[i]-=1,t.s[D]+=1,s||(M=t.H[o],Q==1?t.H[o]=2:t.H[o]=4,(M==129||M==257)&&(V=t.D[0][o],k=e.pq(t,V,false),t.D[0][k]!=V&&(t.D[1][k]=t.D[0][k],t.D[0][k]=V))),t.T=0;}for(p=0,g=0,b=0,r=0;r<t.q;r++)p+=t.s[r],t.s[r]<0&&!e.Pp(t,r)&&(++g,e.Hm(t,r)&&(b-=t.s[r]));if(!A&&p!=0)throw Ut(new T.cC("molecule's overall charges are not balanced"));for(e.bw(t,1),q=0,S=n?p+b:b,u=0;u<t.q;u++)t.s[u]>0&&!e.Op(t,u)&&e.Hm(t,u)&&(L=d.Math.min(e.op(t,u),t.s[u]),L!=0&&S>=L&&(p-=L,q-=L,S-=L,t.s[u]-=L,t.T&=1));if(B=n?p:q,B<0){for(c=j(l.j7,H,6,g,15,1),g=0,i=0;i<t.q;i++)t.s[i]<0&&!e.Pp(t,i)&&(c[g++]=(t.C[i]<<22)+i);for(w.vzb(c,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),h=c.length-1;B<0&&h>=c.length-g;h--)P=c[h]&_1,e.Hm(t,P)&&(L=d.Math.min(-B,-t.s[P]),p+=L,B+=L,t.s[P]+=L,t.T&=1);}return p;},e.Ao=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b;for(n=0;n<t.q;n++)b=e.pm(t,n),b+e.nm(t,n,b,t.C[n])-e.vp(t,n)<=0&&!(t.s[n]==0&&(t.C[n]==5||(r=t.C[n],r==7||r==15||r==33)||(P=t.C[n],P==8||P==16||P==34||P==52||P==84)))&&(t.B[n]=J(t.B[n],-6145));if(t.S)return  false;for(D=false,A=0;A<t.f;A++)if(u=t.e[A]-t.j[A],u>0)for(H1(J(t.B[A],j1),0)&&(L=H1(J(t.B[A],X0),896)?3:H1(J(t.B[A],X0),384)?2:H1(J(t.B[A],128),128)?1:0,o=(b=e.pm(t,A),b+e.nm(t,A,b,t.C[A])-e.vp(t,A)),t.s[A]==0&&H1(J(t.B[A],mA),0)&&t.C[A]!=6&&++o,s=u,s>3-L&&(s=3-L),s>o+u-L&&(s=o+u-L),s>0&&(h=L==0?0:h1(J(t.B[A],X0),s),h=rt(h,l.Ueb((s==3?7:u==2?3:1)<<7)),t.B[A]=J(t.B[A],-1921),t.B[A]=rt(t.B[A],J(X0,h)))),Q=t.j[A];Q<t.e[A];Q++)i=t.k[A][Q],t.H[i]==1&&(t.C[t.i[A][Q]]=-1,t.H[i]=512,D=true);return D&&e.kl(t),D;},e.Bo=function(t,A){var n,P,r,i;if(t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2){for(r=0;r<2;r++)if(n=e.Go(t,A,t.i[A][r]),n!=-1)for(i=0;i<t.j[n];i++)P=t.k[n][i],(t.H[P]==257||t.H[P]==129)&&t.D[0][P]==n&&(t.H[t.k[n][i]]=1);return;}if(t.o[A]==0||t.C[A]>=15)for(r=0;r<t.e[A];r++)P=t.k[A][r],(t.H[P]==257||t.H[P]==129)&&t.D[0][P]==A&&t.D[0][P]==A&&(t.H[P]=1);},e.Co=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h;for(e.bw(t,P?7:1),A.v=null,t.K&&e.On(A,true),o=n.length,r==null&&(r=j(l.j7,H,6,o,15,1)),A.q=0,i=0;i<o;i++)if(n[i]){if(r[i]=e.ll(t,A,i,0,0),t.C[i]==1&&(h=e.vp(t,i),h>1)){for(s=0;s<t.e[i];s++)if(!n[t.i[i][s]]){e.gn(A,r[i],h);break;}}}else r[i]=-1;for(L=j(l.j7,H,6,t.r,15,1),y.Tqb(L,L.length,-1),A.r=0,Q=0;Q<t.r;Q++)D=t.D[0][Q],u=t.D[1][Q],D<o&&u<o&&(n[D]&&n[u]?L[Q]=e.ml(t,A,Q,0,0,r==null?t.D[0][Q]:r[t.D[0][Q]],r==null?t.D[1][Q]:r[t.D[1][Q]],P):t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(n[D]&&(A.s[r[D]]+=t.s[D]<0?1:-1),n[u]&&(A.s[r[u]]+=t.s[u]<0?1:-1)));e.pl(t,A),t.d&&(A.T=0),A.T=0,e.cn(A,1),e.cn(A,2),A.q!=o&&e.On(A,true),e.sq(t,A,o,r,L),e.rq(t,A,o,r),P&&e.Ge(new e.Ve(A),null,false);},e.Do=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h;for(e.bw(t,P?7:1),A.v=null,t.K&&e.On(A,true),r==null&&(r=j(l.j7,H,6,t.q,15,1)),A.q=0,i=0;i<t.q;i++)for(r[i]=-1,L=0;L<t.j[i];L++)if(n[t.k[i][L]]){if(r[i]=e.ll(t,A,i,0,0),t.C[i]==1&&(h=e.vp(t,i),h>1)){for(s=0;s<t.e[i];s++)if(!n[t.k[i][s]]){e.gn(A,r[i],h);break;}}break;}for(Q=j(l.j7,H,6,t.r,15,1),A.r=0,o=0;o<t.r;o++)n[o]?Q[o]=e.ml(t,A,o,0,0,r==null?t.D[0][o]:r[t.D[0][o]],r==null?t.D[1][o]:r[t.D[1][o]],P):(Q[o]=-1,D=t.D[0][o],u=t.D[1][o],r[D]==-1^r[u]==-1&&t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(r[D]!=-1&&(A.s[r[D]]+=t.s[D]<0?1:-1),r[u]!=-1&&(A.s[r[u]]+=t.s[u]<0?1:-1)));return e.pl(t,A),t.d&&(A.T=0),A.T=0,e.cn(A,1),e.cn(A,2),A.q!=t.q&&e.On(A,true),e.sq(t,A,t.q,r,Q),e.rq(t,A,t.q,r),P&&e.Ge(new e.Ve(A),null,false),r;},e.Eo=function(t,A){var n,P,r,i,D,u,o,Q,L,s;if((A&~t.T)!=0&&((t.T&1)==0&&(e.Np(t),e.yo(t),t.T|=1,t.K&&!e.vm(t)&&e.Ao(t)&&(e.Np(t),e.yo(t),t.T|=1)),(A&~t.T)!=0)){if((t.T&-7)!=0){for(P=0;P<t.f;P++)t.u[P]&=-15369;for(i=0;i<t.g;i++)t.F[i]&=-705;if((A&4)==0){e.Jo(t,1),t.T|=2;return;}for(e.Jo(t,7),r=0;r<t.f;r++)for(L=0;L<t.j[r];L++)if(o=t.k[r][L],!(o<t.g&&e.pt(t.p,o)))for(u=t.i[r][L],s=0;s<t.j[u];s++)t.k[u][s]!=o&&t.n[u][s]>1&&(t.C[t.i[u][s]]==6?t.u[r]|=Z1:!e.Up(t,t.k[u][s])&&e.Hm(t,t.i[u][s])&&(t.u[r]|=L0));for(;;){for(Q=false,n=0;n<t.f;n++)if(t.o[n]>0&&(t.u[n]&L0)!=0&&!e.ot(t.p,n)){for(L=0;L<t.j[n];L++)if(t.n[n][L]>1)for(u=t.i[n][L],o=t.k[n][L],s=0;s<t.j[u];s++)t.k[u][s]!=o&&(D=t.i[u][s],(t.u[D]&L0)==0&&(t.u[D]|=L0,Q=true));}if(!Q)break;}}t.T|=6;}},e.Fo=function(t,A){var n,P,r,i,D;if(n=-1,t.o[A]==1){for(i=0;i<t.j[A];i++)if(t.n[A][i]==2){if(P=t.i[A][i],t.j[P]==2&&t.o[P]==2){for(D=0;D<2;D++)if(r=t.i[P][D],r!=A&&t.o[r]==1){n=P;break;}}break;}}return n;},e.Go=function(t,A,n){var P,r;for(P=A;t.j[n]==2&&t.o[n]==2&&n!=P;)r=n,n=t.i[n][0]==A?t.i[n][1]:t.i[n][0],A=r;return n==P?-1:n;},e.Ho=function(t,A){var n;if(t.j[A]==3&&A<t.f&&e.ot(t.p,A)&&(t.p&&A<t.f?e.dt(t.p,A):0)>=5){for(n=0;n<t.j[A];n++)if(e.Vp(t,t.k[A][n]))return t.k[A][n];}return  -1;},e.Io=function(t,A,n,P,r){var i,D,u,o,Q,L;if(e.bw(t,7),!((t.u[A]&V0)==0||n&&!(A<t.f&&e.ot(t.p,A))))for(o=j(l.j7,H,6,t.f,15,1),o[0]=A,P[A]=true,u=0,Q=0;u<=Q;){for(L=0;L<t.j[o[u]];L++)D=t.k[o[u]][L],!r[D]&&(t.F[D]&64)!=0&&(!n||D<t.g&&e.pt(t.p,D))&&(r[D]=true,i=t.i[o[u]][L],P[i]||(P[i]=true,o[++Q]=i));++u;}},e.Jo=function(t,A){var n,P,r,i,D,u,o,Q;for(t.p=new e.Dt(t,A),P=j(l.j7,H,6,t.f,15,1),r=0;r<t.g;r++)e.ft(t.p,r)!=0&&(t.F[r]|=64,++P[t.D[0][r]],++P[t.D[1][r]]);for(n=0;n<t.f;n++)P[n]==2?t.u[n]|=i0:P[n]==3?t.u[n]|=j1:P[n]>3&&(t.u[n]|=V0);for(Q=0;Q<t.p.j.a.length;Q++)for(D=e.ht(t.p,Q),o=e.it(t.p,Q),u=D.length,i=0;i<u;i++)t.u[D[i]]|=8,t.F[o[i]]|=128,t.H[o[i]]==386&&(t.H[o[i]]=2);},e.Ko=function(t){var A,n,P,r,i,D,u,o;for(u=j(l.Feb,ht,6,t.q,16,1),n=0;n<t.q;n++)u[n]=t.C[n]==1&&t.A[n]==0&&t.s[n]==0&&(t.u[n]&cA)==0&&(t.t==null||t.t[n]==null);for(o=j(l.Feb,ht,6,t.q,16,1),D=0;D<t.r;D++){if(P=t.D[0][D],r=t.D[1][D],e.fm(t,D)!=1){u[P]=false,u[r]=false;continue;}o[P]&&(u[P]=false),o[r]&&(u[r]=false),u[P]&&e.Lm(t,r)&&t.C[r]!=13&&(u[P]=false),u[r]&&e.Lm(t,P)&&t.C[P]!=13&&(u[r]=false),o[P]=true,o[r]=true;}for(i=0;i<t.r;i++)u[t.D[0][i]]&&u[t.D[1][i]]&&(u[t.D[0][i]]=false,u[t.D[1][i]]=false);for(A=0;A<t.q;A++)o[A]||(u[A]=false);return u;},e.Lo=function(t,A){return t.e[A];},e.Mo=function(t,A){return t.i[A].length;},e.No=function(t,A){return t.e[A]-e.tp(t,A)+e.op(t,A);},e.Oo=function(t){var A,n;for(e.bw(t,7),A=0,n=0;n<t.p.j.a.length;n++)e.nt(t.p,n)&&++A;return A;},e.Po=function(t,A){var n,P;for(n=0,P=0;P<t.j[A];P++)e.Hm(t,t.i[A][P])&&!e.Cm(t,t.k[A][P])&&++n;return n;},e.Qo=function(t,A){return t.o[A];},e.Ro=function(t,A){return e.bw(t,7),t.o[A]==2&&t.j[A]==2?e.nq(t,A,false):e.pq(t,A,false);},e.So=function(t,A){var n;return n=t.u[A]&V0,n==0?0:n==i0?2:n==j1?3:4;},e.To=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;for(e.bw(t,7),i=j(l.Feb,ht,6,t.g,16,1),s=j(l.Feb,ht,6,t.g,16,1),c=j(l.j7,H,6,t.f,15,1),D=0,u=1;u<t.j[A];u++)if(P=t.k[A][u],(t.F[P]&64)!=0){for(Q=0;Q<u;Q++)if(r=t.k[A][Q],(t.F[r]&64)!=0&&(s[P]=true,s[r]=true,b=e.yp(t,c,t.i[A][u],t.i[A][Q],n-2,null,s),s[P]=false,s[r]=false,b!=-1)){for(o=false,h=j(l.j7,H,6,b,15,1),e.zp(t,c,h,b),L=0;L<b;L++)i[h[L]]||(i[h[L]]=true,o=true);o&&++D;}}return D;},e.Uo=function(t,A){return t.p&&A<t.f?e.dt(t.p,A):0;},e.Vo=function(t,A){return A?(e.bw(t,1),e.Wl(t,t.f,t.g,e.Qk)):e.Wl(t,t.q,t.r,e.Qk);},e.Wo=function(t){var A,n,P,r,i,D,u,o;for(e.bw(t,1),u=j(l.i7,w0,6,t.f,15,1),P=j(l.j7,H,6,t.f,15,1),o=0;o<t.f;o++){for(P[0]=o,r=j(l.j7,H,6,t.f,15,1),r[o]=1,n=0,i=0;n<=i;){for(D=0;D<t.j[P[n]];D++)A=t.i[P[n]][D],r[A]==0&&(r[A]=r[P[n]]+1,P[++i]=A,u[o]+=r[A]-1);++n;}u[o]/=i;}return u;},e.Xo=function(t,A,n){var P;for(P=0;P<t.i[A].length;P++)if(t.i[A][P]==n)return t.k[A][P];return  -1;},e.Yo=function(t,A){return t.p&&A<t.g?e.ft(t.p,A):0;},e.Zo=function(t,A,n){return t.i[A][n];},e.$o=function(t,A){return t.j[A];},e._o=function(t,A,n){return t.k[A][n];},e.ap=function(t,A,n){return t.n[A][n];},e.bp=function(t,A){var n,P;if(n=0,t.K)for(P=0;P<t.j[A];P++)e.Jm(t,t.i[A][P])&&++n;return n;},e.cp=function(t,A){return t.e[A]-t.j[A];},e.dp=function(t,A,n,P,r){var i,D,u,o;if(D=t.e[A],t.o[A]!=0||A<t.f&&e.ot(t.p,A)||t.j[A]<3||D>4)return  false;for(o=j(l.Feb,ht,6,4,16,1),u=0;u<D;u++){if(i=3.9269908169872414-P[u],d.Math.abs(Qr-i%k0)>.0872664675116539||(r[u]=3&Rt(i/k0),o[r[u]]))return  false;if(o[r[u]]=true,(r[u]&1)==0){if(t.H[t.k[A][n[u]]]!=1)return  false;}else if(!e.Sm(t,t.k[A][n[u]],A))return  false;}return o[0]&&o[2];},e.ep=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h;if((t.p&&A<t.f?e.dt(t.p,A):0)>24||(i=t.e[A],r==null&&(r=j(l.j7,H,6,i,15,1)),!e.dp(t,A,n,P,r)))return 3;for(o=-1,Q=0;Q<i;Q++)if((r[Q]&1)==1){if(D=t.H[t.k[A][n[Q]]],o!=-1&&o!=D)return 3;o=D;}return L=d.Math.abs(r[0]-r[1])==2?1:0,u=r[L]-r[L+1],h=d.Math.abs(u)==3^r[L]<r[L+1],s=i==3||(r[3]&1)==1,h^s^o==129?1:2;},e.fp=function(t,A,n){var P,r,i,D;for(D=D=j(l.Feb,ht,6,t.q,16,1),i=e.gp(t,A,n,D),r=j(l.j7,H,6,i,15,1),i=0,P=0;P<t.q;P++)D[P]&&(r[i++]=P);return r;},e.gp=function(t,A,n,P){var r,i,D,u,o,Q,L;for(e.bw(t,1),P==null&&(P=j(l.Feb,ht,6,t.q,16,1)),o=j(l.j7,H,6,t.q,15,1),o[0]=A,P[A]=true,D=0,Q=0,u=1;D<=Q;){for(i=n?e.Mo(t,o[D]):t.e[o[D]],L=0;L<i;L++)r=t.i[o[D]][L],r<P.length&&!P[r]&&(o[++Q]=r,P[r]=true,++u);++D;}return u;},e.hp=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h;for(e.bw(t,1),i=0;i<t.q;i++)A[i]=-1;for(Q=0,r=0;r<t.q;r++)if(A[r]==-1&&(!n||(t.u[r]&n0)!=0)){for(A[r]=Q,L=j(l.j7,H,6,t.q,15,1),L[0]=r,o=0,s=0;o<=s;){for(u=P?e.Mo(t,L[o]):t.e[L[o]],h=0;h<u;h++)D=t.i[L[o]][h],A[D]==-1&&(!n||(t.u[D]&n0)!=0)&&(L[++s]=D,A[D]=Q);++o;}++Q;}return Q;},e.ip=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h;for(e.bw(t,1),i=0;i<t.q;i++)A[i]=-1;for(Q=0,r=0;r<t.q;r++)if(A[r]==-1){for(A[r]=Q,L=j(l.j7,H,6,t.q,15,1),L[0]=r,o=0,s=0;o<=s;){for(u=P?e.Mo(t,L[o]):t.e[L[o]],h=0;h<u;h++)D=t.i[L[o]][h],A[D]==-1&&!n[t.k[L[o]][h]]&&(L[++s]=D,A[D]=Q);++o;}++Q;}return Q;},e.jp=function(t,A){var n;return n=e.pm(t,A),n+e.nm(t,A,n,t.C[A])-e.vp(t,A);},e.kp=function(t,A){var n,P,r,i,D,u;for(i=j(l.j7,H,6,t.q,15,1),P=0;P<t.q;P++)i[P]=P;r=t.q;do--r;while(r>=0&&A[r]);for(n=0;n<r;n++)if(A[i[n]]){u=i[n],i[n]=i[r],i[r]=u,D=A[n],A[n]=A[r],A[r]=D;do--r;while(A[r]);}return i;},e.lp=function(t){return e.kp(t,e.Ko(t));},e.mp=function(t,A,n){var P,r,i;if(r=e.vp(t,A),r-=e.nm(t,A,r,t.C[A]),n&&(r-=t.e[A]-t.j[A]),i=e.mo(t.C[A]),r<=i[0])return  -1;for(P=1;P<i.length;P++)if(i[P]>=r)return i[P];return r;},e.np=function(t){var A,n;if(t.K)return 0;for(e.bw(t,1),n=0,A=0;A<t.f;A++)n+=e.op(t,A);return n;},e.op=function(t,A){var n,P,r,i,D,u,o;if(t.K&&H1(J(t.B[A],j1),0)||!e.Aq(t,A))return 0;if(t.C[A]==0||T.kjb("*",t.t==null||t.t[A]==null?null:T.ijb((i=t.t[A],S1.Mmb(),i))))return t.e[A]==0?1:0;for(e.bw(t,1),u=0,r=0;r<t.e[A];r++)u+=t.n[A][r];if(t.K){for(n=1,P=0;P<t.j[A];P++)t.H[t.k[A][P]]==8&&++n;u+=n>>1;}if(u-=e.nm(t,A,u,t.C[A]),D=((t.u[A]&cA)>>>27)-1,D==-1)for(o=e.mo(t.C[A]),D=o[0],P=1;D<u&&P<o.length;P++)D=o[P];return d.Math.max(0,D-u);},e.pp=function(t,A){var n,P,r,i,D;if(!t.K||t.v==null||t.v[A]==null)return e.qp(t,A,t.C[A]);for(D=0,P=t.v[A],r=0,i=P.length;r<i;++r)n=P[r],D=d.Math.max(D,e.qp(t,A,n));return D;},e.qp=function(t,A,n){var P,r,i,D,u;if(i=e.vp(t,A),P=e.nm(t,A,i,n),D=((t.u[A]&cA)>>>27)-1,D==-1){for(u=e.mo(t.C[A]),r=0;r<u.length-1&&i>u[r]+P;)++r;D=u[r];}return D+P-i;},e.rp=function(t,A){return t.i[A].length-t.e[A];},e.sp=function(t){var A,n,P,r;for(e.bw(t,1),r=0,A=0;A<t.q;A++)P=t.A[A]!=0?t.A[A]:e.Pk[t.C[A]],r+=P+e.op(t,A)*e.Pk[1],t.C[A]>=171&&t.C[A]<=190&&(n=t.e[A],n>2&&(r-=(n-2)*e.Pk[1]));return r;},e.tp=function(t,A){var n,P;for(n=t.j[A],P=0;P<t.j[A];P++)t.C[t.i[A][P]]==1&&--n;return n;},e.up=function(t,A){return t.j[A]-e.bp(t,A);},e.vp=function(t,A){var n,P,r,i,D,u;for(t.$b(1),D=false,P=false,u=0,r=0;r<t.e[A];r++)(!t.K||!e.Jm(t,t.i[A][r]))&&(i=t.n[A][r],u+=i,i>1&&(D=true),n=t.k[A][r],t.H[n]==8&&(P=true));return P&&!D&&++u,u;},e.wp=function(t,A,n){var P,r,i;for(r=0,i=0;i<t.j[A];i++)P=t.i[A][i],P!=n&&t.j[P]>2&&++r;return r;},e.xp=function(t,A){var n,P,r,i,D,u,o,Q,L,s;if(e.to(t.C[A]))return 0;if(t.C[A]==1)return t.j[A]==0?0:e.ro(t.C[t.i[A][0]])?-1:1;if(s=t.s[A],o=t.e[A]-e.tp(t,A)+e.op(t,A),t.C[A]!=1&&t.C[A]!=15&&(e.ro(t.C[A])?s+=o:s-=o),t.j[A]==0)return s;for(L=0;L<t.j[A];L++)i=t.i[A][L],n=e.Yp(t,t.k[A][L])?1:t.n[A][L],e.ro(t.C[A])?e.ro(t.C[i])||(s+=n):e.ro(t.C[i])?s-=n:t.C[A]!=t.C[i]&&(e.Xs(t.C[A]).a<e.Xs(t.C[i]).a?s+=n:s-=n);if(A<t.f&&e.tt(t.p,A)){for(P=0,r=0,u=0,Q=0;Q<t.j[A];Q++)e.Yp(t,t.k[A][Q])&&(D=t.C[t.i[A][Q]],D==6?++r:e.qo(D)?++u:++P);t.C[A]==6?P>1?--s:u>1&&++s:e.Hm(t,A)?P+r>1&&--s:u+r>1&&++s;}return s;},e.yp=function(t,A,n,P,r,i,D){var u,o,Q,L,s,h,b,c,g;if(n==P)return A[0]=n,0;for(t.$b(1),L=j(l.j7,H,6,t.q,15,1),Q=j(l.j7,H,6,t.q,15,1),g=j(l.j7,H,6,t.q,15,1),Q[0]=n,L[n]=1,o=0,s=0;o<=s&&L[Q[o]]<=r;){for(c=Q[o],h=0;h<t.e[c];h++)if(D==null||D.length<=t.k[c][h]||!D[t.k[c][h]]){if(u=t.i[c][h],u==P){for(b=L[c],A[b]=u,A[--b]=c;b>0;)A[b-1]=g[A[b]],--b;return L[c];}L[u]==0&&(i==null||i.length<=u||!i[u])&&(Q[++s]=u,L[u]=L[c]+1,g[u]=c);}++o;}return  -1;},e.zp=function(t,A,n,P){var r,i;for(e.bw(t,1),r=0;r<P;r++)for(i=0;i<t.e[A[r]];i++)if(t.i[A[r]][i]==A[r+1]){n[r]=t.k[A[r]][i];break;}},e.Ap=function(t,A,n){var P,r,i,D,u,o;if(A==n)return 0;for(e.bw(t,1),D=j(l.j7,H,6,t.q,15,1),i=j(l.j7,H,6,t.q,15,1),i[0]=A,D[A]=1,r=0,u=0;r<=u;){for(o=0;o<t.e[i[r]];o++){if(P=t.i[i[r]][o],P==n)return D[i[r]];D[P]==0&&(i[++u]=P,D[P]=D[i[r]]+1);}++r;}return  -1;},e.Bp=function(t,A,n,P,r){var i,D,u,o,Q,L;if(A==n)return 0;for(t.$b(1),o=j(l.j7,H,6,t.q,15,1),u=j(l.j7,H,6,t.q,15,1),u[0]=A,o[A]=1,D=0,Q=0;D<=Q&&o[u[D]]<=P;){for(L=0;L<t.e[u[D]];L++){if(i=t.i[u[D]][L],i==n)return o[u[D]];o[i]==0&&(r==null||r.length<=i||!r[i])&&(u[++Q]=i,o[i]=o[u[D]]+1);}++D;}return  -1;},e.Cp=function(t,A){return t.e[A]-t.j[A]+e.op(t,A);},e.Dp=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c;for(c=j(l.j7,H,6,16,15,1),n=j(l.h7,Ft,6,16,15,1),i=j(l.h7,Ft,6,2,15,1),P=0,L=0;L<2;L++)for(r=t.D[L][A],s=0;s<t.j[r];s++)if(o=t.k[r][s],o!=A){if(P==4)return 0;u=t.i[r][s],c[P]=16,t.o[u]!=0&&(c[P]+=(t.u[u]&V0)!=0?1:4),(t.F[A]&64)!=0&&(t.F[o]&64)!=0&&(h=e.lt(t.p,A,o),h!=-1&&(c[P]+=e.nt(t.p,h)?64:6)),n[P++]=e.no(t.J[r].a,t.J[r].b,t.J[u].a,t.J[u].b);}for(i[0]=e.Yl(t,t.D[0][A],t.D[1][A]),i[0]<0?(i[1]=i[0]+A1,D=false):(i[1]=i[0],i[0]=i[1]-A1,D=true),b=0,Q=0;Q<P;Q++)n[Q]>i[0]&&n[Q]<i[1]?b-=c[Q]:b+=c[Q];return D?-b:b;},e.Ep=function(t){return e.bw(t,7),t.p;},e.Fp=function(t){var A,n,P,r,i,D,u,o,Q;for(Q=0,e.bw(t,7),P=0;P<t.g;P++)if(e.fm(t,P)==1&&(t.F[P]&64)==0){for(u=true,D=0;D<2;D++){if(A=t.D[D][P],t.j[A]==1){u=false;break;}if(t.C[A]==7&&!(A<t.f&&e.ot(t.p,A))){for(n=t.D[1-D][P],o=0;o<t.j[n];o++)if(r=t.i[n][o],i=t.k[n][o],i!=P&&e.fm(t,i)>1&&!(r<t.f&&e.ot(t.p,r))&&e.Hm(t,r)){u=false;break;}}}u&&!e.cq(t,P)&&++Q;}return Q;},e.Gp=function(t,A,n){var P,r,i,D,u,o;for(r=0,i=0;i<t.j[A];i++)if(P=t.i[A][i],P!=n){for(D=0,u=0;u<t.j[P];u++)o=t.i[P][u],o!=A&&e.Up(t,t.k[P][u])&&t.j[o]>2&&++D;D==2&&++r;}return r;},e.Hp=function(t,A){var n,P,r,i;for(n=t.e[A],i=j(l.j7,H,6,n,15,1),r=0;r<n;r++)i[r]=(t.i[A][r]<<16)+r;for(w.vzb(i,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),P=0;P<n;P++)i[P]&=n1;return i;},e.Ip=function(t,A){var n,P;if(e.bw(t,1),t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2){for(n=0;n<2;n++)for(P=0;P<t.e[t.i[A][n]];P++)if(e.Sm(t,t.k[t.i[A][n]][P],t.i[A][n]))return t.k[t.i[A][n]][P];}else for(n=0;n<t.e[A];n++)if(e.Sm(t,t.k[A][n],A))return t.k[A][n];return  -1;},e.Jp=function(t,A,n){return e.fm(t,A)!=1?0:16-t.e[n]+16*d.Math.max(0,d.Math.min(9,t.p&&A<t.g?e.ft(t.p,A):0)-2)+((t.H[A]&384)==0||t.D[0][A]!=n?LA:0)+((t.u[n]&3)==0?Z1:0)+(t.C[n]==1?j1:0)+(t.e[n]==1?i0:0)+((t.F[A]&64)!=0?0:512)+(t.C[n]!=6?256:0);},e.Kp=function(t,A,n,P,r,i){var D,u,o,Q,L;for(e.bw(t,1),r&&(e.jl(r),r.d=null,r.K=false),o=j(l.j7,H,6,t.q,15,1),P==null?P=j(l.Feb,ht,6,t.q,16,1):y.Xqb(P,P.length),o[0]=A,o[1]=n,P[A]=true,P[n]=true,u=1,Q=1;u<=Q;){for(L=0;L<t.e[o[u]];L++){if(D=t.i[o[u]][L],D==A&&u!=1)return  -1;P[D]||(P[D]=true,o[++Q]=D);}++u;}return r&&(i==null&&(i=j(l.j7,H,6,P.length,15,1)),e.Co(t,r,P,false,i),e.fl(r,i[A],0,0,-1,0)),P[A]=false,Q;},e.Lp=function(t,A,n){var P,r,i,D,u,o;for(e.bw(t,1),i=j(l.j7,H,6,t.f,15,1),o=j(l.Feb,ht,6,t.f,16,1),i[0]=A,i[1]=n,o[A]=true,o[n]=true,r=1,D=1;r<=D;){for(u=0;u<t.j[i[r]];u++){if(P=t.i[i[r]][u],P==A&&r!=1)return  -1;o[P]||(o[P]=true,i[++D]=P);}++r;}return D;},e.Mp=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b;if(e.fm(t,n)!=2&&!(n<t.g&&e.pt(t.p,n))||(b=t.F[n]&3,b!=1&&b!=2))return  -1;for(o=0;o<2;o++){for(P=t.D[o][n],r=t.D[1-o][n],h=-1,D=false,L=0;L<t.j[P];L++)i=t.i[P][L],i!=r&&(i==A?D=true:h=i);if(D){for(s=-1,u=-1,Q=0;Q<t.j[r];Q++)i=t.i[r][Q],i!=P&&(s==-1?s=i:i>s?u=i:(u=s,s=i));return t.j[P]==2?t.j[r]==2?b==2?s:-1:b==2?s:u:t.j[r]==2?b==2^A<h?-1:s:b==2^A<h?u:s;}}return  -1;},e.Np=function(t){var A,n,P,r,i,D,u,o,Q,L;D=e.Ko(t),o=t.q;do--o;while(o>=0&&D[o]);for(A=0;A<o;A++)if(D[A]){e.Zn(t,A,o),L=D[A],D[A]=D[o],D[o]=L;do--o;while(D[o]);}if(t.f=o+1,t.q==t.f){t.g=t.r;return;}for(u=j(l.Feb,ht,6,t.r,16,1),i=0;i<t.r;i++)n=t.D[0][i],P=t.D[1][i],(D[n]||D[P])&&(u[i]=true);Q=t.r;do--Q;while(Q>=0&&u[Q]);for(r=0;r<Q;r++)if(u[r]){e.$n(t,r,Q),u[r]=false;do--Q;while(u[Q]);}t.g=Q+1;},e.Op=function(t,A){var n;for(n=0;n<t.j[A];n++)if(t.s[t.i[A][n]]<0)return  true;return  false;},e.Pp=function(t,A){var n;for(n=0;n<t.j[A];n++)if(t.s[t.i[A][n]]>0)return  true;return  false;},e.Qp=function(t,A){var n;return n=t.C[A],n==3||n==11||n==19||n==37||n==55||n==87;},e.Rp=function(t,A){return (t.u[A]&Z1)!=0;},e.Sp=function(t,A){var n,P,r,i,D,u;for(e.bw(t,1),D=0;D<2;D++)if(n=t.D[D][A],t.C[n]==7){for(P=t.D[1-D][A],u=0;u<t.j[P];u++)if(r=t.i[P][u],i=t.k[P][u],(t.C[r]==7||t.C[r]==8||t.C[r]==16)&&e.fm(t,i)>=2)return  true;}return  false;},e.Tp=function(t,A){return A<t.f&&e.ot(t.p,A);},e.Up=function(t,A){return A<t.g&&e.pt(t.p,A);},e.Vp=function(t,A){var n,P,r,i,D,u;return t.H[A]!=1||A<t.g&&e.pt(t.p,A)||(t.F[A]&64)!=0&&(t.p&&A<t.g?e.ft(t.p,A):0)<7||(n=t.D[0][A],!(n<t.f&&e.ot(t.p,n))||(t.p&&n<t.f?e.dt(t.p,n):0)<5)||(P=t.D[1][A],!(P<t.f&&e.ot(t.p,P))||(t.p&&P<t.f?e.dt(t.p,P):0)<5)?false:(r=e.wp(t,n,P),i=e.wp(t,P,n),(t.p&&n<t.f?e.dt(t.p,n):0)>5&&(t.p&&P<t.f?e.dt(t.p,P):0)>5?r+i>2:(D=e.Gp(t,n,P),u=e.Gp(t,P,n),r==2&&u>=1||i==2&&D>=1||D==2&&(i>=1||u>=1)||u==2&&(r>=1||D>=1)));},e.Wp=function(t,A){return t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2&&t.C[A]<=7;},e.Xp=function(t,A){var n;return n=t.C[A],n==8||n==16||n==34||n==52||n==84;},e.Yp=function(t,A){return A<t.g&&(e.ut(t.p,A)||t.H[A]==8);},e.Zp=function(t,A){var n;return n=t.C[A],n==4||n==12||n==20||n==38||n==56||n==88;},e.$p=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g;if(t.C[A]!=7||t.j[A]==4)return  false;if(A<t.f&&e.ot(t.p,A)||t.o[A]!=0||pt(J(t.B[A],IA),0))return  true;if(t.s[A]==1)return  false;for(o=0;o<t.j[A];o++)if(e.Yo(t,t.k[A][o])==3)return  false;for(D=0,Q=0;Q<t.j[A];Q++)t.n[A][Q]==1&&(P=t.C[t.i[A][Q]],(P==8||P==9||P==17)&&++D);if(D==0){for(u=0;u<t.j[A];u++)if(r=t.i[A][u],t.o[r]!=0){if(r<t.f&&e.ot(t.p,r)){if((t.p&&r<t.f?e.dt(t.p,r):0)>=5){for(g=0,h=0;h<t.j[r];h++)c=t.i[r][h],c!=A&&e.tp(t,c)>=3&&++g;if(b=e.tp(t,A),g==2&&b>=2||g==1&&b==3)continue;}return !n||!e.eq(t,A);}for(s=0;s<t.j[r];s++)if(t.n[r][s]==2||e.Up(t,t.k[r][s]))return !n||!e.eq(t,A);}}if(D<2)for(u=0;u<t.j[A];u++){for(r=t.i[A][u],L=false,i=false,s=0;s<t.j[r];s++)t.i[r][s]!=A&&(t.n[r][s]!=1&&(t.C[t.i[r][s]]==7||t.C[t.i[r][s]]==8||t.C[t.i[r][s]]==16)&&(L=true),t.n[r][s]==1&&t.C[t.i[r][s]]==7&&(i=true));if(L&&(!i||D==0))return !n||!e.eq(t,A);}return  false;},e._p=function(t,A){var n;return n=t.C[A],n==9||n==17||n==35||n==53||n==85;},e.aq=function(t,A){return A<t.f&&e.vt(t.p,A);},e.bq=function(t,A){var n;return n=t.C[A],n==7||n==15||n==33;},e.cq=function(t,A){var n,P,r,i,D,u;if(e.fm(t,A)!=1)return  false;for(i=0;i<2;i++){for(n=t.D[i][A],u=t.D[1-i][A];t.o[n]==2&&t.j[n]==2&&t.C[n]<10;)for(D=0;D<2;D++)if(P=t.i[n][D],P!=u){if(t.j[P]==1||(r=t.k[n][D],e.fm(t,r)==1&&r<A))return  true;u=n,n=P;break;}if(t.j[n]==1)return  true;}return  false;},e.dq=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c,g,B,k,M;if(h=j(l.Feb,ht,6,t.g,16,1),h[r]=true,b=j(l.j7,H,6,11,15,1),c=e.yp(t,b,P,A,10,null,h),c==-1)return  false;for(D=1;!e.qt(t.p,n,b[D]);)++D;if(i=c-D,u=b[D],M=e.kt(t.p,n),M==6&&i==2&&D==3&&e.So(t,b[1])>=3){for(k=e.ht(t.p,n),s=0;s<6;s++)if(A==k[s]){if(B=e.Ct(t.p,n,u==k[e.Ct(t.p,n,s+2)]?s-2:s+2),g=k[B],L=t.u[g]&V0,(L==0?0:L==i0?2:L==j1?3:4)>=3&&e.Bp(t,b[1],g,2,null)==2)return  true;break;}}if(o=t.o[u]==1||u<t.f&&e.ot(t.p,u)||e.$p(t,u,false),Q=!o&&t.C[u]==7&&t.s[u]!=1,i==1)return !o&&!Q&&M<=4&&D<=3;switch(M){case 4:if(!o&&!Q&&D<=4)return  true;break;case 5:if(Q){if(D<=3)return  true;}else if(!o&&D<=4)return  true;break;case 6:if(i==2){if(o){if(D<=4)return  true;}else if(!Q&&D<=3)return  true;}else if(i==3){if(o){if(D<=6)return  true;}else if(D<=4)return  true;}break;case 7:if(i==3&&D<=3)return  true;}return  false;},e.eq=function(t,A){var n,P,r,i,D;if(A<t.f&&e.ot(t.p,A)||t.o[A]!=0||pt(J(t.B[A],IA),0)||(P=t.u[A]&V0,(P==0?0:P==i0?2:P==j1?3:4)<3)||(D=t.p&&A<t.f?e.dt(t.p,A):0,D>7))return  false;for(i=0;i<t.p.j.a.length&&!(e.kt(t.p,i)==D&&e.qt(t.p,i,A));)++i;if(i>=i0&&i==t.p.j.a.length)return  false;for(r=0;r<t.j[A];r++)if(n=t.k[A][r],!e.rt(t.p,i,n)&&e.dq(t,A,i,t.i[A][r],n))return  true;return  false;},e.fq=function(t,A){return (t.u[A]&V0)!=0;},e.gq=function(t,A){return (t.F[A]&64)!=0;},e.hq=function(t,A){return t.C[A]==1&&t.A[A]==0&&t.s[A]==0&&(t.u[A]&cA)==0&&(t.t==null||t.t[A]==null);},e.iq=function(t,A){return (t.u[A]&8)!=0;},e.jq=function(t,A){return (t.F[A]&128)!=0;},e.kq=function(t,A){return (t.u[A]&L0)!=0;},e.lq=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(e.bw(t,1),e.mq(t),Q=false,n=0;n<t.f;n++)if(t.C[n]==7&&t.s[n]==0){if(s=e.vp(t,n),s==4){for(L=0;L<t.j[n];L++)if(u=t.i[n][L],t.n[n][L]==1&&t.C[u]==8&&t.j[u]==1&&t.s[u]==0){Q=true,++t.s[n],--t.s[u];break;}}else if(s==5)for(L=0;L<t.j[n];L++){if(u=t.i[n][L],o=t.k[n][L],t.n[n][L]==2&&t.C[u]==8){Q=true,++t.s[n],--t.s[u],t.H[o]=1;break;}if(t.n[n][L]==3&&t.C[u]==7){Q=true,++t.s[n],--t.s[u],t.H[o]=2;break;}}}for(D=false,i=0;i<t.g;i++)for(L=0;L<2;L++)if(e.Hm(t,t.D[L][i])){A=t.D[1-L][i],r=t.C[A],(r==3||r==11||r==19||r==37||r==55||r==87||(P=t.C[A],P==4||P==12||P==20||P==38||P==56||P==88))&&(e.fm(t,i)==1?(++t.s[A],--t.s[t.D[L][i]],t.H[i]=512,D=true):t.H[i]==16&&(t.H[i]=512,D=true));break;}return D&&(e.kl(t),Q=true),Q&&(t.T=0),Q;},e.mq=function(t){var A;for(A=0;A<t.g;A++)if(t.H[A]==8)return e.Ge(new e.Ve(t),null,false);return  false;},e.nq=function(t,A,n){var P,r,i,D,u,o,Q,L;for(Q=-1,r=0,u=0;u<2;u++)for(P=t.i[A][u],o=0;o<t.e[P];o++)i=t.i[P][o],i!=A&&(D=t.k[P][o],L=e.Jp(t,D,i),r<L&&(!n||!(t.H[D]==257||t.H[D]==129))&&(r=L,Q=D));return Q;},e.oq=function(t,A){var n,P,r,i,D,u,o,Q;for(o=-1,P=0,D=0;D<2;D++)for(n=t.D[D][A],u=0;u<t.e[n];u++)r=t.i[n][u],r!=t.D[1-D][A]&&(i=t.k[n][u],Q=e.Jp(t,i,r),P<Q&&(P=Q,o=i));return o;},e.pq=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;for(P=t.e[A],r=j(l.h7,Ft,6,P,15,1),b=0;b<P;b++)r[b]=e.Yl(t,A,t.i[A][b]);for(c=1;c<P;c++)for(g=0;g<c;g++)if(i=d.Math.abs(e.oo(r[c],r[g])),i<.08||i>J8){for(D=0,u=0,B=0;B<P;B++)B!=c&&B!=g&&(D+=d.Math.abs(Lt.H3(r[c],r[B])),u+=d.Math.abs(Lt.H3(r[g],r[B])));if(Q=D<u?t.k[A][c]:t.k[A][g],e.fm(t,Q)==1&&(!n||!(t.H[Q]==257||t.H[Q]==129)))return Q;}for(k=-1,o=0,h=0;h<P;h++)L=t.i[A][h],s=t.k[A][h],M=e.Jp(t,s,L),o<M&&(!n||!(t.H[s]==257||t.H[s]==129))&&(o=M,k=s);return k;},e.qq=function(t,A){var n,P,r,i,D,u;for(e.bw(t,A?15:1),t.q=t.f,t.r=t.g,u=j(l.Feb,ht,6,t.f,16,1),r=0;r<t.f;r++)t.e[r]!=t.j[r]&&(n=e.mp(t,r,false),u[r]=t.e[r]==t.j[r]+1&&e.Rm(t,t.k[r][t.j[r]]),t.e[r]=t.j[r],n!=-1&&(D=e.mp(t,r,true),n!=D&&(i=((t.u[r]&cA)>>>27)-1,(i==-1||i<n)&&e.gn(t,r,n))));if(A)for(P=0;P<t.f;P++)u[P]&&e.wq(t,P);t.T=0;},e.rq=function(t,A,n,P){var r,i,D,u;for(e.bw(A,1),i=0;i<n;i++)P[i]!=-1&&t.C[i]!=1&&t.e[i]!=e.Lo(A,P[i])&&(r=e.mp(t,i,false),r!=-1&&(u=e.mp(A,P[i],false),r!=u&&(D=e.zl(A,P[i]),(D==-1||D<r)&&e.gn(A,P[i],r))));},e.sq=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c,g;for(u=0;u<n;u++)if(P[u]!=-1&&t.e[u]>3&&(t.u[u]&S0)!=0){for(g=0,h=-1,s=-1,L=0;L<t.e[u];L++)r[t.k[u][L]]!=-1?++g:t.n[u][L]==1&&e.Rm(t,t.k[u][L])&&t.D[0][t.k[u][L]]==u&&(h=t.k[u][L],s=t.i[u][L]);if(h!=-1&&g>=3){for(i=e.no(t.J[u].a,t.J[u].b,t.J[s].a,t.J[s].b),b=10,c=-1,Q=0;Q<t.e[u];Q++)t.n[u][Q]==1&&(!e.Rm(t,t.k[u][Q])||t.D[0][t.k[u][Q]]==u)&&r[t.k[u][Q]]!=-1&&(D=d.Math.abs(e.oo(i,e.Yl(t,u,t.i[u][Q]))),b>D&&(b=D,c=t.k[u][Q]));c!=-1&&(o=r[c],e.Mn(A,o,t.H[c]==257?129:257),t.D[0][c]!=u&&(e.Dn(A,1,o,P[t.D[0][c]]),e.Dn(A,0,o,P[u])));}}},e.tq=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I;if(t.j[A]!=2||t.n[A][0]!=2||t.n[A][1]!=2||t.j[t.i[A][0]]<2||t.j[t.i[A][1]]<2||t.o[t.i[A][0]]!=1||t.o[t.i[A][1]]!=1){e.un(t,A,0,false);return;}for(S=-1,q=-1,p=-1,B=-1,i=0,s=0;s<2;s++)for(P=t.i[A][s],c=0;c<t.e[P];c++)D=t.i[P][c],D!=A&&(u=t.k[P][c],V=e.Jp(t,u,D),i<V&&(i=V,q=D,S=u,p=P,B=t.i[A][1-s]));if(q!=-1){for(h=0;h<2;h++)for(P=t.i[A][h],c=0;c<t.e[P];c++)D=t.i[P][c],u=t.k[P][c],D!=A&&t.D[0][u]==P&&(t.H[u]=1);for(t.D[1][S]!=q&&(t.D[0][S]=t.D[1][S],t.D[1][S]=q),o=St,b=0;b<t.j[p];b++)D=t.i[p][b],D!=A&&o>D&&(o=D);for(k=j(l.j7,H,6,2,15,1),M=0,L=0;L<t.j[B];L++)D=t.i[B][L],D!=A&&(k[M++]=D);n=e.no(t.J[A].a,t.J[A].b,t.J[B].a,t.J[B].b),r=0,M==2?(k[0]>k[1]&&(I=k[0],k[0]=k[1],k[1]=I),Q=e.oo(n,e.Yl(t,B,k[0])),g=e.oo(n,e.Yl(t,B,k[1])),r=Q-g):r=e.oo(n,e.Yl(t,B,k[0])),r<0^(t.u[A]&3)==1^o==q?t.H[S]=257:t.H[S]=129;}},e.uq=function(t,A,n,P){var r,i,D,u,o,Q;if(r=t.e[A],u=j(l.j7,H,6,r,15,1),Q=e.ep(t,A,n,P,u),Q==3)return  false;for(D=(t.u[A]&3)==Q?257:129,o=0;o<r;o++)(u[o]&1)==1&&(i=t.k[A][n[o]],t.H[i]=D,t.D[0][i]!=A&&(t.D[1][i]=t.D[0][i],t.D[0][i]=A));return  true;},e.vq=function(t,A){t.T|=248&(8|A);},e.wq=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B;if(e.Bo(t,A),!((t.u[A]&3)==0||(t.u[A]&3)==3)){if(t.o[A]==2&&t.j[A]==2){e.tq(t,A);return;}if(t.j[A]<3||t.j[A]>4){e.un(t,A,0,false);return;}for(n=t.e[A],c=false,D=0;D<n;D++)if(e.fm(t,t.k[A][D])==1){c=true;break;}if(c){for(g=e.Hp(t,A),P=j(l.h7,Ft,6,n,15,1),u=0;u<n;u++)P[u]=e.Yl(t,t.i[A][g[u]],A);for(o=0;o<n;o++)t.D[0][t.k[A][o]]==A&&e.fm(t,t.k[A][o])==1&&(t.H[t.k[A][o]]=1);if(!((t.p&&A<t.f?e.dt(t.p,A):0)<=24&&e.uq(t,A,g,P))&&(h=e.pq(t,A,true),h!=-1)){for(t.D[0][h]!=A&&(t.D[1][h]=t.D[0][h],t.D[0][h]=A),b=-1,Q=0;Q<n;Q++)if(h==t.k[A][g[Q]]){b=Q;break;}for(B=v(F(l.j7,2),E1,7,0,[v(F(l.j7,1),H,6,15,[2,1,2,1]),v(F(l.j7,1),H,6,15,[1,2,2,1]),v(F(l.j7,1),H,6,15,[1,1,2,2]),v(F(l.j7,1),H,6,15,[2,1,1,2]),v(F(l.j7,1),H,6,15,[2,2,1,1]),v(F(l.j7,1),H,6,15,[1,2,1,2])]),i=1;i<n;i++)P[i]<P[0]&&(P[i]+=I1);if(n==3){switch(L=false,b){case 0:L=P[1]<P[2]&&P[2]-P[1]<A1||P[1]>P[2]&&P[1]-P[2]>A1;break;case 1:L=P[2]-P[0]>A1;break;case 2:L=P[1]-P[0]<A1;}r=(t.u[A]&3)==1^L?257:129;}else s=0,P[1]<=P[2]&&P[2]<=P[3]?s=0:P[1]<=P[3]&&P[3]<=P[2]?s=1:P[2]<=P[1]&&P[1]<=P[3]?s=2:P[2]<=P[3]&&P[3]<=P[1]?s=3:P[3]<=P[1]&&P[1]<=P[2]?s=4:P[3]<=P[2]&&P[2]<=P[1]&&(s=5),r=(t.u[A]&3)==1^B[s][b]==1?129:257;t.H[h]=r;}}}},e.xq=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I;if(!((t.F[A]&3)==0||(t.F[A]&3)==3||!e.Vp(t,A))){for(S=-1,p=-1,q=-1,M=-1,r=0,s=0;s<2;s++)for(P=t.D[s][A],c=0;c<t.e[P];c++)u=t.k[P][c],u!=A&&e.fm(t,u)==1&&(D=t.i[P][c],V=e.Jp(t,u,D),r<V&&(r=V,p=D,S=u,q=P,M=t.D[1-s][A]));if(p!=-1){for(h=0;h<2;h++)for(c=0;c<t.e[t.D[h][A]];c++)u=t.k[t.D[h][A]][c],u!=A&&e.fm(t,u)==1&&(t.H[u]=1);for(t.D[1][S]!=p&&(t.D[0][S]=t.D[1][S],t.D[1][S]=p),o=St,b=0;b<t.j[q];b++)D=t.i[q][b],t.k[q][b]!=A&&o>D&&(o=D);for(B=j(l.j7,H,6,2,15,1),k=0,L=0;L<t.j[M];L++)t.k[M][L]!=A&&(B[k++]=t.i[M][L]);i=e.no(t.J[q].a,t.J[q].b,t.J[M].a,t.J[M].b),n=0,k==2?(B[0]>B[1]&&(I=B[0],B[0]=B[1],B[1]=I),Q=e.oo(i,e.Yl(t,M,B[0])),g=e.oo(i,e.Yl(t,M,B[1])),n=Q-g):n=e.oo(i,e.Yl(t,M,B[0])),n<0^(t.F[A]&3)==2^o==p?t.H[S]=257:t.H[S]=129;}}},e.yq=function(t){var A,n,P,r;for(e.bw(t,7),P=0;P<t.g;P++)(t.H[P]==257||t.H[P]==129)&&(t.H[P]=1);for(A=0;A<t.f;A++)e.wq(t,A);for(r=0;r<t.g;r++)e.xq(t,r);for(n=0;n<t.g;n++)t.H[n]==2&&(t.F[n]&3)==3&&(t.H[n]=386);},e.zq=function(t,A){var n,P,r,i,D,u,o,Q,L,s;if(u=j(l.j7,H,6,t.q,15,1),D=e.hp(t,u,false,A),D<=1)return null;for(o=j(l.j7,H,6,D,15,1),P=0;P<t.f;P++)++o[u[P]];for(L=0,s=o[0],Q=1;Q<D;Q++)s<o[Q]&&(s=o[Q],L=Q);for(n=0;n<t.q;n++)u[n]!=L&&(t.C[n]=-1);for(i=0;i<t.r;i++)(!A&&t.H[i]==16||u[t.D[0][i]]!=L)&&(t.H[i]=512);r=e.kl(t),t.T=0;try{e.zo(t,!0,!0);}catch(h){if(h=M1(h),!Kt(h,19))throw Ut(h);}return r;},e.Aq=function(t,A){return (t.u[A]&cA)!=0?true:t.C[A]<=1?false:e.Nm(t,A)||t.C[A]==13||t.C[A]>=171;},e.Bq=function(t){var A,n,P,r,i,D,u,o,Q,L;for(i=e.Wl(t,t.q,t.r,e.Qk),D=i*i/16,P=1;P<t.q;P++)for(r=0;r<P;r++)if(o=t.J[r].a-t.J[P].a,Q=t.J[r].b-t.J[P].b,L=t.J[r].c-t.J[P].c,o*o+Q*Q+L*L<D)throw Ut(new T.cC("The distance between two atoms is too close."));for(e.bw(t,1),A=0,n=0;n<t.f;n++){if(e.vp(t,n)>(u=e.pm(t,n),u+e.nm(t,n,u,t.C[n])))throw Ut(new T.cC("atom valence exceeded"));A+=t.s[n];}if(A!=0)throw Ut(new T.cC("unbalanced atom charge"));},e.Cq=function(t){var A,n;if(t.K)for(e.bw(t,7),A=0;A<t.f;A++)(t.u[A]&V0)!=0&&(H1(J(t.B[A],120),8)&&(t.B[A]=J(t.B[A],-9)),t.B[A]=J(t.B[A],-4294967297)),A<t.f&&e.ot(t.p,A)?t.B[A]=J(t.B[A],{l:4194297,m:_1,h:1048571}):pt(J(t.B[A],2),0)&&(t.B[A]=J(t.B[A],-5)),(pt(J(t.B[A],kA),0)||pt(J(t.B[A],EA),0)&&H1(J(t.B[A],QA),0))&&H1(J(t.B[A],120),8)&&(t.B[A]=J(t.B[A],-9)),t.s[A]!=0&&(t.B[A]=J(t.B[A],-234881025)),e.vp(t,A)==(n=e.pm(t,A),n+e.nm(t,A,n,t.C[A]))&&(t.B[A]=J(t.B[A],-4063233),t.B[A]=J(t.B[A],-17042430230529),t.B[A]=J(t.B[A],-1921),t.B[A]=J(t.B[A],-114689)),pt(J(t.B[A],r0),0)&&t.w[A]!=0&&e.an(t,t.w[A]);},e.Dq=function(t){var A,n,P;if(t.K)for(e.bw(t,7),A=0;A<t.g;A++)P=Pe(t.G[A]&127),(A<t.g&&(e.ut(t.p,A)||t.H[A]==8))&(t.G[A]&8)!=0&&(t.G[A]&=-9,--P),P!=0&&(n=t.H[A]&127,n==1?t.G[A]|=1:n==2?t.G[A]|=2:n==4?t.G[A]|=4:n==32?t.G[A]|=32:n==64?t.G[A]|=64:n==16?t.G[A]|=16:n==8&&(t.G[A]|=8));},e.Eq=function(t,A,n){var P;return P=e.ao(t,A,n),P&&n==386&&(e.bw(t,7),P=P&(t.F[A]&128)==0),P;},e.Fq=function(){e.eo.call(this);},e.Gq=function(t,A){e.fo.call(this,t,A);},e.Hq=function(t){e.fo.call(this,t?t.M:256,t?t.N:256),t&&e.ol(t,this);},N(114,88,{114:1,88:1,4:1}),U.$b=function(t){e.Eo(this,t);},U.Zb=function(t){return e.Yp(this,t);},U.f=0,U.g=0,l.K7=O(114),e.Kq=function(t,A){return d.Math.pow(10,d.Math.log(2e3)*d.Math.LOG10E*t/(A-1)-1);},e.Lq=function(t,A){var n,P;for(n=A,P=0;A!=0;)t.d==0&&(t.f=(t.c[++t.e]&63)<<11,t.d=6),P|=(q1&t.f)>>16-n+A,t.f<<=1,--A,--t.d;return P;},e.Mq=function(t,A,n){t.d=6,t.e=n,t.c=A,t.f=(A[t.e]&63)<<11;},e.Nq=function(t,A){var n,P,r,i;return P=A/2|0,r=t>=P,r&&(t-=P),i=A/32,n=i*t/(P-t),r?-n:n;},e.Oq=function(t){var A,n,P,r;if(t.g.K)for(A=0;A<t.g.r;A++)P=e.hm(t.g,A),(P&127)!=0&&(n=-1,r=0,(P&16)!=0&&(n=16,++r),(P&64)!=0&&(n=64,++r),(P&32)!=0&&(n=32,++r),(P&4)!=0&&(n=4,++r),(P&2)!=0&&(n=2,++r),(P&8)!=0&&(n=8,++r),(P&1)!=0&&(n=1,++r),n!=-1&&(e.Mn(t.g,A,n),r==1&&e.Ln(t.g,A,127,false)));},e.Pq=function(t,A){var n;return A==null||w.aAb(A).length==0?null:e.Rq(t,w.Czb((n=A,S1.Mmb(),n)));},e.Qq=function(t,A,n){var P,r;return A==null?null:e.Sq(t,w.Czb((r=A,S1.Mmb(),r)),n==null?null:w.Czb((P=n,P)),0);},e.Rq=function(t,A){var n;if(A==null||A.length==0)return null;for(n=2;n<A.length-2;n++)if(A[n]==32)return e.Sq(t,A,A,n+1);return e.Sq(t,A,null,0);},e.Sq=function(t,A,n,P){var r,i,D,u,o;return A==null?null:(e.Mq(t,A,0),r=e.Lq(t,4),u=e.Lq(t,4),r>8&&(r=u),i=e.Lq(t,r),D=e.Lq(t,u),o=new e.uw(i,D),e.Vq(t,o,A,n,P),o);},e.Tq=function(t,A,n){var P,r,i,D;if(n==null||w.aAb(n).length==0){e.Uq(t,A,null,null);return;}P=T.rjb(n,Y1(32)),P>0&&P<w.aAb(n).length-1?e.Uq(t,A,w.Czb((i=(w.Wzb(0,P,w.aAb(n).length),w.aAb(n).substr(0,P)),S1.Mmb(),i)),w.Czb((D=(w.Xzb(P+1,w.aAb(n).length+1),w.aAb(n).substr(P+1)),D))):e.Uq(t,A,w.Czb((r=n,S1.Mmb(),r)),null);},e.Uq=function(t,A,n,P){if(n==null||n.length==0){e.jl(A),A.d=null;return;}e.Vq(t,A,n,P,0);},e.Vq=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt,b1,u1,Ht,gt,w1,x1,L1,O1,U1,l1,G1,a1,p0,J1,o1,D0,T0,d0,CA,jA,eA,nA,C0,MA,Z0,M0,YA,Be,mt,Wt,BA,de,me,ke,i8,f8,D8,u8,o8,l8,Q8,L8,SP,VP,U8,y8,s8,h8,T8,b8,H0,c8,w8,g8,j8,B8,d8,m8,k8,F8,v8,oA,p8,re,R8,Xe,Pu,ru,iu,fu,Du,$t,C8,uu,ou,lu,XA,Qu,lA,_e,xP,Cn,Lu,PA,zP,Uu,Mn,yu,Hn,su,qn;if(e.jl(A),A.d=null,!(n==null||0>=n.length)){if(t.g=A,zP=8,P!=null&&(r<0||r>=P.length)&&(P=null),e.Mq(t,n,0),u=e.Lq(t,4),I=e.Lq(t,4),u>8&&(zP=u,u=I),u==0){e.On(t.g,e.Lq(t,1)==1);return;}for(o=e.Lq(t,u),Q=e.Lq(t,I),Du=e.Lq(t,u),lu=e.Lq(t,u),uu=e.Lq(t,u),yt=e.Lq(t,u),b=0;b<o;b++)e.Uk(t.g,6);for(T0=0;T0<Du;T0++)e.Cn(t.g,e.Lq(t,u),7);for(d0=0;d0<lu;d0++)e.Cn(t.g,e.Lq(t,u),8);for(mt=0;mt<uu;mt++)e.Cn(t.g,e.Lq(t,u),e.Lq(t,8));for(l8=0;l8<yt;l8++)e.jn(t.g,e.Lq(t,u),e.Lq(t,4)-8);for(wt=1+Q-o,Bt=e.Lq(t,4),V=0,e.zn(t.g,0,0),e.An(t.g,0,0),e.Bn(t.g,0,0),jt=P!=null&&P[r]>=39,PA=0,Mn=0,Hn=0,qn=0,qt=false,Zt=false,jt&&(P.length>2*o-2&&P[2*o-2]==39||P.length>3*o-3&&P[3*o-3]==39?(Zt=true,qt=P.length==3*o-3+9,oA=qt?3*o-3:2*o-2,S=86*(P[oA+1]-40)+P[oA+2]-40,PA=d.Math.pow(10,S/2e3-1),oA+=2,Uu=86*(P[oA+1]-40)+P[oA+2]-40,Mn=d.Math.pow(10,Uu/1500-1),oA+=2,yu=86*(P[oA+1]-40)+P[oA+2]-40,Hn=d.Math.pow(10,yu/1500-1),qt&&(oA+=2,su=86*(P[oA+1]-40)+P[oA+2]-40,qn=d.Math.pow(10,su/1500-1))):qt=P.length==3*o-3),t._b()&&qt&&(P=null,jt=false),H0=1;H0<o;H0++){if(Ht=e.Lq(t,Bt),Ht==0){jt&&(e.zn(t.g,H0,e.Ql(t.g,0)+8*(P[H0*2-2]-83)),e.An(t.g,H0,e.Rl(t.g,0)+8*(P[H0*2-1]-83)),qt&&e.Bn(t.g,H0,e.Sl(t.g,0)+8*(P[2*o-3+H0]-83))),++wt;continue;}V+=Ht-1,jt&&(e.zn(t.g,H0,e.Ql(t.g,V)+P[H0*2-2]-83),e.An(t.g,H0,e.Rl(t.g,V)+P[H0*2-1]-83),qt&&e.Bn(t.g,H0,e.Sl(t.g,V)+(P[2*o-3+H0]-83))),e.Wk(t.g,V,H0,1);}for(d8=0;d8<wt;d8++)e.Wk(t.g,e.Lq(t,u),e.Lq(t,u),1);for(p8=j(l.Feb,ht,6,Q,16,1),it=0;it<Q;it++)switch(Qt=e.Lq(t,2),Qt){case 0:p8[it]=true;break;case 2:e.Mn(t.g,it,2);break;case 3:e.Mn(t.g,it,4);}for(D=e.Lq(t,u),m8=0;m8<D;m8++)if(h=e.Lq(t,u),zP==8)XA=e.Lq(t,2),XA==3?(e.on(t.g,h,1,0),e.un(t.g,h,1,false)):e.un(t.g,h,XA,false);else switch(XA=e.Lq(t,3),XA){case 4:e.un(t.g,h,1,false),e.on(t.g,h,1,e.Lq(t,3));break;case 5:e.un(t.g,h,2,false),e.on(t.g,h,1,e.Lq(t,3));break;case 6:e.un(t.g,h,1,false),e.on(t.g,h,2,e.Lq(t,3));break;case 7:e.un(t.g,h,2,false),e.on(t.g,h,2,e.Lq(t,3));break;default:e.un(t.g,h,XA,false);}for(zP==8&&e.Lq(t,1)==0&&(t.g.L=true),i=e.Lq(t,I),k8=0;k8<i;k8++)if(z=e.Lq(t,I),e.im(t.g,z)==1)switch(XA=e.Lq(t,3),XA){case 4:e.Jn(t.g,z,1,false),e.Gn(t.g,z,1,e.Lq(t,3));break;case 5:e.Jn(t.g,z,2,false),e.Gn(t.g,z,1,e.Lq(t,3));break;case 6:e.Jn(t.g,z,1,false),e.Gn(t.g,z,2,e.Lq(t,3));break;case 7:e.Jn(t.g,z,2,false),e.Gn(t.g,z,2,e.Lq(t,3));break;default:e.Jn(t.g,z,XA,false);}else e.Jn(t.g,z,e.Lq(t,2),false);for(e.On(t.g,e.Lq(t,1)==1),s=null,Cn=null,C8=0;e.Lq(t,1)==1;)switch(It=C8+e.Lq(t,4),It){case 0:for($t=e.Lq(t,u),F8=0;F8<$t;F8++)h=e.Lq(t,u),e.vn(t.g,h,j1,true);break;case 1:for($t=e.Lq(t,u),v8=0;v8<$t;v8++)h=e.Lq(t,u),iu=e.Lq(t,8),e.tn(t.g,h,iu);break;case 2:for($t=e.Lq(t,I),CA=0;CA<$t;CA++)e.Lq(t,I);break;case 3:for($t=e.Lq(t,u),jA=0;jA<$t;jA++)h=e.Lq(t,u),e.vn(t.g,h,Z1,true);break;case 4:for($t=e.Lq(t,u),eA=0;eA<$t;eA++)h=e.Lq(t,u),xP=h1(Mt(e.Lq(t,4)),3),e.vn(t.g,h,xP,true);break;case 5:for($t=e.Lq(t,u),nA=0;nA<$t;nA++)h=e.Lq(t,u),L=h1(Mt(e.Lq(t,2)),1),e.vn(t.g,h,L,true);break;case 6:for($t=e.Lq(t,u),C0=0;C0<$t;C0++)h=e.Lq(t,u),e.vn(t.g,h,1,true);break;case 7:for($t=e.Lq(t,u),MA=0;MA<$t;MA++)h=e.Lq(t,u),J1=h1(Mt(e.Lq(t,4)),7),e.vn(t.g,h,J1,true);break;case 8:for($t=e.Lq(t,u),Z0=0;Z0<$t;Z0++){for(h=e.Lq(t,u),M=e.Lq(t,4),B=j(l.j7,H,6,M,15,1),re=0;re<M;re++)k=e.Lq(t,8),B[re]=k;e.pn(t.g,h,B);}break;case 9:for($t=e.Lq(t,I),M0=0;M0<$t;M0++)z=e.Lq(t,I),xP=e.Lq(t,2)<<7,e.Ln(t.g,z,xP,true);break;case 10:for($t=e.Lq(t,I),YA=0;YA<$t;YA++)z=e.Lq(t,I),Dt=e.Lq(t,5),e.Ln(t.g,z,Dt,true);break;case 11:for($t=e.Lq(t,u),Be=0;Be<$t;Be++)h=e.Lq(t,u),e.vn(t.g,h,L0,true);break;case 12:for($t=e.Lq(t,I),Wt=0;Wt<$t;Wt++)z=e.Lq(t,I),lt=e.Lq(t,8)<<9,e.Ln(t.g,z,lt,true);break;case 13:for($t=e.Lq(t,u),BA=0;BA<$t;BA++)h=e.Lq(t,u),Qu=h1(Mt(e.Lq(t,3)),14),e.vn(t.g,h,Qu,true);break;case 14:for($t=e.Lq(t,u),de=0;de<$t;de++)h=e.Lq(t,u),fu=h1(Mt(e.Lq(t,5)),17),e.vn(t.g,h,fu,true);break;case 15:case 31:C8+=16;break;case 16:for($t=e.Lq(t,u),me=0;me<$t;me++)h=e.Lq(t,u),_e=h1(Mt(e.Lq(t,3)),22),e.vn(t.g,h,_e,true);break;case 17:for($t=e.Lq(t,u),ke=0;ke<$t;ke++)h=e.Lq(t,u),e.gn(t.g,h,e.Lq(t,4));break;case 18:for($t=e.Lq(t,u),ru=e.Lq(t,4),i8=0;i8<$t;i8++){for(h=e.Lq(t,u),Nt=e.Lq(t,ru),R8=j(l.f7,Y,6,Nt,15,1),re=0;re<Nt;re++)R8[re]=e.Lq(t,7)<<24>>24;e.mn(t.g,h,T.ijb((Xe=R8,S1.Mmb(),Xe)));}break;case 19:for($t=e.Lq(t,u),f8=0;f8<$t;f8++)h=e.Lq(t,u),ct=h1(Mt(e.Lq(t,3)),25),e.vn(t.g,h,ct,true);break;case 20:for($t=e.Lq(t,I),D8=0;D8<$t;D8++)z=e.Lq(t,I),_e=e.Lq(t,3)<<17,e.Ln(t.g,z,_e,true);break;case 21:for($t=e.Lq(t,u),u8=0;u8<$t;u8++)h=e.Lq(t,u),e.wn(t.g,h,e.Lq(t,2)<<4);break;case 22:for($t=e.Lq(t,u),o8=0;o8<$t;o8++)h=e.Lq(t,u),e.vn(t.g,h,IA,true);break;case 23:for($t=e.Lq(t,I),Q8=0;Q8<$t;Q8++)z=e.Lq(t,I),e.Ln(t.g,z,hA,true);break;case 24:for($t=e.Lq(t,I),L8=0;L8<$t;L8++)z=e.Lq(t,I),L=e.Lq(t,2)<<21,e.Ln(t.g,z,L,true);break;case 25:for(SP=0;SP<o;SP++)e.Lq(t,1)==1&&e.xn(t.g,SP,true);break;case 26:for($t=e.Lq(t,I),s=j(l.j7,H,6,$t,15,1),VP=0;VP<$t;VP++)s[VP]=e.Lq(t,I);break;case 27:for($t=e.Lq(t,u),U8=0;U8<$t;U8++)h=e.Lq(t,u),e.vn(t.g,h,r0,true);break;case 28:for($t=e.Lq(t,I),y8=0;y8<$t;y8++)e.Mn(t.g,e.Lq(t,I),16);break;case 29:for($t=e.Lq(t,u),s8=0;s8<$t;s8++)h=e.Lq(t,u),p0=h1(Mt(e.Lq(t,2)),30),e.vn(t.g,h,p0,true);break;case 30:for($t=e.Lq(t,u),h8=0;h8<$t;h8++)h=e.Lq(t,u),_e=h1(Mt(e.Lq(t,7)),32),e.vn(t.g,h,_e,true);break;case 32:for($t=e.Lq(t,u),T8=0;T8<$t;T8++)h=e.Lq(t,u),Lu=h1(Mt(e.Lq(t,2)),44),e.vn(t.g,h,Lu,true);break;case 33:for($t=e.Lq(t,u),b8=0;b8<$t;b8++)h=e.Lq(t,u),O1=h1(Mt(e.Lq(t,5)),39),e.vn(t.g,h,O1,true);break;case 34:for($t=e.Lq(t,u),c8=0;c8<$t;c8++)h=e.Lq(t,u),e.vn(t.g,h,en,true);break;case 35:for($t=e.Lq(t,I),w8=0;w8<$t;w8++)z=e.Lq(t,I),e.Ln(t.g,z,Le,true);break;case 36:for($t=e.Lq(t,I),g8=0;g8<$t;g8++)z=e.Lq(t,I),ut=e.Lq(t,2)<<5,e.Ln(t.g,z,ut,true);break;case 37:for($t=e.Lq(t,I),j8=0;j8<$t;j8++)z=e.Lq(t,I),ut=e.Lq(t,1)==0?32:64,e.Mn(t.g,z,ut);break;case 38:for($t=e.Lq(t,u),bt=e.Lq(t,3),Cn=j(l.j7,H,6,o,15,1),B8=0;B8<$t;B8++)h=e.Lq(t,u),xt=e.Lq(t,bt),Cn[h]=xt;break;case 39:for($t=e.Lq(t,u),D0=0;D0<$t;D0++)h=e.Lq(t,u),ou=h1(Mt(e.Lq(t,4)),47),e.vn(t.g,h,ou,true);}if(e.Ge(new e.Ve(t.g),p8,false),s!=null)for(tt=s,ot=0,G=tt.length;ot<G;++ot)z=tt[ot],e.Mn(t.g,z,e.im(t.g,z)==2?4:2);if(P==null&&!t.i&&n.length>t.e+1&&(n[t.e+1]==32||n[t.e+1]==9)&&(P=n,r=t.e+2),P!=null)try{if(P[r]==33||P[r]==35){for(e.Mq(t,P,r+1),qt=e.Lq(t,1)==1,Zt=e.Lq(t,1)==1,lA=2*e.Lq(t,4),Z=1<<lA,G1=0,z=0,c=1;c<o;c++)z<Q&&e.Zl(t.g,1,z)==c?(G1=e.Zl(t.g,0,z++),l1=1):(G1=0,l1=8),b1=l1*(e.Lq(t,lA)+1-(Z>>1)),u1=l1*(e.Lq(t,lA)+1-(Z>>1)),e.zn(t.g,c,e.Ql(t.g,G1)+b1),e.An(t.g,c,e.Rl(t.g,G1)+u1),qt&&e.Bn(t.g,c,e.Sl(t.g,G1)+l1*(e.Lq(t,lA)+1-(Z>>1)));if(P[r]==35){for(o1=0,a1=j(l.j7,H,6,o,15,1),g=0;g<o;g++)o1+=a1[g]=e.op(t.g,g);for(h=0;h<o;h++)for(D0=0;D0<a1[h];D0++)J1=e.Uk(t.g,1),e.Wk(t.g,h,J1,1),e.zn(t.g,J1,e.Ql(t.g,h)+(e.Lq(t,lA)+1-(Z>>1))),e.An(t.g,J1,e.Rl(t.g,h)+(e.Lq(t,lA)+1-(Z>>1))),qt&&e.Bn(t.g,J1,e.Sl(t.g,h)+(e.Lq(t,lA)+1-(Z>>1))),Cn!=null&&(Cn[h]&1<<D0)!=0&&e.xn(t.g,J1,!0);o+=o1,Q+=o1;}if(q=qt?1.5:(e.Rk(),e.Rk(),e.Qk),p=e.Wl(t.g,o,Q,q),Zt)for(PA=e.Kq(e.Lq(t,lA),Z),Mn=PA*e.Nq(e.Lq(t,lA),Z),Hn=PA*e.Nq(e.Lq(t,lA),Z),qt&&(qn=PA*e.Nq(e.Lq(t,lA),Z)),l1=PA/p,h=0;h<o;h++)e.zn(t.g,h,Mn+l1*e.Ql(t.g,h)),e.An(t.g,h,Hn+l1*e.Rl(t.g,h)),qt&&e.Bn(t.g,h,qn+l1*e.Sl(t.g,h));else for(PA=1.5,l1=PA/p,h=0;h<o;h++)e.zn(t.g,h,l1*e.Ql(t.g,h)),e.An(t.g,h,l1*e.Rl(t.g,h)),qt&&e.Bn(t.g,h,l1*e.Sl(t.g,h));}else if(qt&&!Zt&&PA==0&&(PA=1.5),PA!=0&&t.g.r!=0){for(p=0,z=0;z<t.g.r;z++)gt=e.Ql(t.g,e.Zl(t.g,0,z))-e.Ql(t.g,e.Zl(t.g,1,z)),w1=e.Rl(t.g,e.Zl(t.g,0,z))-e.Rl(t.g,e.Zl(t.g,1,z)),x1=qt?e.Sl(t.g,e.Zl(t.g,0,z))-e.Sl(t.g,e.Zl(t.g,1,z)):0,p+=d.Math.sqrt(gt*gt+w1*w1+x1*x1);for(p/=t.g.r,U1=PA/p,h=0;h<t.g.q;h++)e.zn(t.g,h,e.Ql(t.g,h)*U1+Mn),e.An(t.g,h,e.Rl(t.g,h)*U1+Hn),qt&&e.Bn(t.g,h,e.Sl(t.g,h)*U1+qn);}}catch(dA){if(dA=M1(dA),Kt(dA,19))L1=dA,T.MB(L1,(T.Zjb(),T.Yjb),""),""+L1+T.ijb((Pu=n,S1.Mmb(),Pu))+T.ijb((Xe=P,Xe)),P=null,qt=false;else throw Ut(dA);}if(Et=P!=null&&!qt,e.Oq(t),Et||t._b())for(e.bw(t.g,7),z=0;z<t.g.g;z++)e.fm(t.g,z)==2&&!e.jq(t.g,z)&&e.gm(t.g,z)==0&&e.Kn(t.g,z);if(e.vq(t.g,0),!Et&&t._b())try{t.ac(t.g),Et=!0;}catch(dA){if(dA=M1(dA),Kt(dA,19))L1=dA,T.MB(L1,(T.Zjb(),T.Yjb),""),""+L1+T.ijb((Xe=n,S1.Mmb(),Xe));else throw Ut(dA);}Et?(e.yq(t.g),e.ow(t.g)):qt||e.vq(t.g,0);}},e.Wq=function(t,A){var n,P,r,i,D,u;if(!(A.length<=0||A[0]<64))for(e.Mq(t,A,0),u=e.Lq(t,4),r=e.Lq(t,1)==1,i=e.Lq(t,1)==1,n=0;n<t.g.f;n++)D=e.Lq(t,u),P=r,r&&i&&(P=e.Lq(t,1)==1),e.rn(t.g,n,D,P);},e.Xq=function(){},N(312,1,{},e.Xq),U._b=function(){return  false;},U.ac=function(t){throw Ut(new T.cC("Unexpected request to invent coordinates. Check source code logic!"));},U.d=0,U.e=0,U.f=0,U.i=false,l.L7=O(312),e.$q=function(t){},e._q=function(t){e.$q(this),this.b=t;},N(50,312,{},e._q),U._b=function(){return this.b;},U.ac=function(t){var A;A=new X.CA(this.a),A.g=new y.Evb(78187493520),X.jA(A,t);},U.a=2,U.b=false,l.M7=O(50),e.cr=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k;r=A.a,b=A.g,Q=pt(J(e.Ol(t.k,r),1),0),i=e.Jl(t.k,r),s=i!=null?e.jr(t,r,i,P):Q?"*":e.Il(t.k,r),!Q&&i==null&&e.Tp(t.k,r)&&(t.j&4)==0&&(e.Qo(t.k,r)!=0||e.zl(t.k,r)==-1&&e.Pl(t.k,r)==0)&&(s=w.aAb(s).toLowerCase()),A.f&&(n.a+="("),b!=-1&&e.gr(t,e.Xo(t.k,A.a,A.g),A.g,n),D=e.Bl(t.k,r),D==0&&(t.j&1)!=0&&(u=J(e.Ol(t.k,r),mA),et(u,oe)==0?D=-1:et(u,ue)==0&&(D=1)),L=e.Ml(t.k,r),h=(t.j&2)!=0?e.Ll(t.k,r):0,c=(t.j&1)!=0?e.pr(t,r,P):null,k=!Q&&!e.wr(e.Tl(t.k,r))||i!=null||(e.Nl(t.k,r)==1||e.Nl(t.k,r)==2)&&(B=e.Hl(t.k,r)-1,!(B!=-1&&t.i[B][e.Gl(t.k,r)]<=1))&&(e.Tl(t.k,r)!=7||e.Bl(t.k,r)>0)||e.Tp(t.k,r)&&e.Qo(t.k,r)==0&&(t.j&4)==0||D!=0||L!=0||h!=0||e.zl(t.k,r)!=-1||e.Pl(t.k,r)!=0||c!=null,k&&(n.a+="["),L!=0&&(n.a+=L),n.a+=""+s,(e.Nl(t.k,r)==1||e.Nl(t.k,r)==2)&&(g=e.Hl(t.k,r)-1,!(g!=-1&&t.i[g][e.Gl(t.k,r)]<=1))&&(e.Tl(t.k,r)!=7||e.Bl(t.k,r)>0)&&T.Sjb(n,e.or(t,r,b)),(t.j&1)==0&&k&&(o=e.Cp(t.k,r),o==1?n.a+="H":o>1&&(n.a+="H"+o)),D!=0&&(n.a+=String.fromCharCode(D>0?43:45),d.Math.abs(D)>1&&T.Pjb(n,d.Math.abs(D))),c!=null&&(n.a+=""+c),h!=0&&(n.a+=":",n.a+=h),k&&(n.a+="]"),e.hr(t,A,n),A.e&&(n.a+=")");},e.dr=function(t,A,n){var P,r,i,D,u;for(i=true,u=y.xj(t.g,A).a,D=e.sr(t,u);D!=-1;)P=e.Zo(t.k,u,D),r=e._o(t.k,u,D),D=e.sr(t,P),e.fr(t,new e.$u(P,r,u,n&&i,n&&D==-1),++A),u=P,i=false;},e.er=function(t,A,n){var P,r,i,D;for(r=A,i=0,D=r.length;i<D;++i)P=r[i],t.f[P&1073741823]=(P&x0)!=0^n?2:1;},e.fr=function(t,A,n){y.rj(t.g,n,A),t.b[A.a]=true,A.g!=-1&&(t.c[e.Xo(t.k,A.a,A.g)]=true);},e.gr=function(t,A,n,P){var r,i,D,u,o;o=w.aAb(P.a).length,t.f[A]!=0&&T.Ojb(P,t.f[A]==1?47:92),t.j==1&&(r=e.hm(t.k,A)&127,r!=0&&((r&1)!=0&&t.f[A]==0&&(P.a+="-"),(r&2)!=0&&(w.aAb(P.a).length!=o&&(P.a+=","),P.a+="="),(r&4)!=0&&(w.aAb(P.a).length!=o&&(P.a+=","),P.a+="#"),(r&32)!=0&&(w.aAb(P.a).length!=o&&(P.a+=","),P.a+="$"),(r&64)!=0&&(w.aAb(P.a).length!=o&&(P.a+=","),P.a+="$"),(r&8)!=0&&(w.aAb(P.a).length!=o&&(P.a+=","),P.a+=":"),(r&16)!=0&&(w.aAb(P.a).length!=o&&(P.a+=","),T.Sjb(P,e.Lm(t.k,n)?"<-":"->")))),o==w.aAb(P.a).length&&(!e.Up(t.k,A)||(t.j&4)!=0)&&(i=e.im(t.k,A)&127,i==1?e.Tp(t.k,e.Zl(t.k,0,A))&&e.Tp(t.k,e.Zl(t.k,1,A))&&(t.j&4)==0&&t.f[A]==0&&(P.a+="-"):i==2?P.a+="=":i==4?P.a+="#":i==32||i==64?P.a+="$":i==8?P.a+=":":i==16&&T.Sjb(P,e.Lm(t.k,n)?"<-":"->")),t.j==1&&(D=o==w.aAb(P.a).length?"":";",u=e.hm(t.k,A)&384,u==256?P.a+=D+"@":u==128&&(P.a+=D+"!@"));},e.hr=function(t,A,n){var P,r,i;if(A.c!=null)for(r=0;r<A.c.length;r++)for(i=0;i<e.$o(t.k,A.a);i++)A.c[r]==e.Zo(t.k,A.a,i)&&(P=e._o(t.k,A.a,i),A.d[r]||e.gr(t,P,A.a,n),t.e[P]>9&&(n.a+="%"),T.Pjb(n,t.e[P]));},e.ir=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt;for(ct=new y.Mj(),k=new y.Bqb(t.g);k.a<k.c.a.length;)if(B=y.Aqb(k),B.g!=-1&&(p=e.Xo(t.k,B.a,B.g),!e.Vp(t.k,p)&&!e.jq(t.k,p)&&(e.gm(t.k,p)==1||e.gm(t.k,p)==2))){if(Dt=y.xj(t.g,t.q[B.g]),n=j(l.j7,H,6,e.$o(t.k,B.a)+e.$o(t.k,Dt.a)-2,15,1),Z=0,lt=false,Dt.g!=-1)n[Z++]=Dt.b;else {for(q=-1,yt=-1,S=St,tt=0;tt<e.$o(t.k,Dt.a);tt++)s=e.Zo(t.k,Dt.a,tt),s!=B.a&&(q==-1?(q=tt,S=t.q[s]):S<t.q[s]?yt=tt:(yt=q,q=tt));yt==-1?(it=e.Zo(t.k,Dt.a,q),Qt=e._o(t.k,Dt.a,q),n[Z++]=Qt|(e.ur(t,Dt.a,it)?x0:0)):(h=e.Zo(t.k,Dt.a,q),c=e._o(t.k,Dt.a,q),b=e.Zo(t.k,Dt.a,yt),g=e._o(t.k,Dt.a,yt),n[Z++]=c|(e.ur(t,Dt.a,h)?x0:0),n[Z++]=g|(e.ur(t,Dt.a,b)?0:x0));}if(e.$o(t.k,Dt.a)==3&&Dt.g!=-1){for(tt=0;tt<e.$o(t.k,Dt.a);tt++)if(s=e.Zo(t.k,Dt.a,tt),s!=Dt.g&&s!=B.a){P=e._o(t.k,Dt.a,tt),n[Z++]=P|(e.ur(t,Dt.a,s)?x0:0),s<Dt.g&&(lt=!lt);break;}}for(e.gm(t.k,p)==2&&(lt=!lt),z=0;z<e.$o(t.k,B.a);z++)if(o=e.Zo(t.k,B.a,z),o!=B.g){if(I=lt,e.$o(t.k,B.a)==3){for(G=0;G<e.$o(t.k,B.a);G++)if(s=e.Zo(t.k,B.a,G),s!=B.g&&s!=o){s<o&&(I=!I);break;}}e.Fm(t.k,p)&&(V=e.tg(t.d,p),t.n[V]||(t.n[V]=true,t.o[V]=I),t.o[V]&&(I=!I)),Q=e.Xo(t.k,B.a,o),n[Z++]=Q|(I^e.ur(t,B.a,o)?0:x0);}w.szb(ct.a,n);}for(t.f=j(l.j7,H,6,t.k.g,15,1),ct.a.length!=0&&e.er(t,y.Ej(ct,0),false);ct.a.length!=0;){for(wt=ct.a.length,z=ct.a.length-1;z>=0;z--){for(n=(w.Pzb(z,ct.a.length),ct.a[z]),ut=0,ot=false,L=false,i=n,D=0,u=i.length;D<u;++D)r=i[D],A=r&1073741823,t.f[A]!=0&&(M=(r&x0)!=0^t.f[A]==2,ut==0?ot=M:ot!=M&&(L=true),++ut);ut!=0&&(n=y.Ej(ct,z),L||e.er(t,n,ot));}wt==ct.a.length&&e.er(t,y.Ej(ct,0),false);}},e.jr=function(t,A,n,P){var r,i,D,u,o,Q,L;for(L=w.aAb(P.a).length,0<L?P.a=T.Bjb(P.a,0,0):0>L&&(P.a+=T.Hjb(j(l.g7,e0,6,-L,15,1))),o=e.Tp(t.k,A)&&(t.j&4)==0,i=n,D=0,u=i.length;D<u;++D)r=i[D],w.aAb(P.a).length!=0&&(P.a+=","),Q=(e.Rk(),e.Lk)[r],T.Sjb(P,o?w.aAb(Q).toLowerCase():Q);return P.a;},e.kr=function(t){var A,n,P,r,i,D,u,o;if(!t.k||t.k.q==0)return "";for(e.bw(t.k,15),t.d=new e.Og(t.k,129),r=t.d.T,t.o=j(l.Feb,ht,6,r+1,16,1),t.n=j(l.Feb,ht,6,r+1,16,1),t.i=y1(l.j7,[E1,H],[7,6],15,[2,32],2),A=0;A<t.k.f;A++)o=e.Hl(t.k,A)-1,o!=-1&&++t.i[o][e.Gl(t.k,A)];for(e.nr(t),e.lr(t),e.ir(t),P=new T.Ujb(),n=new T.Ujb(),i=true,u=new y.Bqb(t.g);u.a<u.c.a.length;)D=y.Aqb(u),D.g==-1&&(i?i=false:P.a+="."),e.cr(t,D,P,n);return P.a;},e.lr=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(s=new y.Bqb(t.g);s.a<s.c.a.length;){for(Q=y.Aqb(s),n=0,i=0;i<e.$o(t.k,Q.a);i++)t.c[e._o(t.k,Q.a,i)]||++n;if(n!=0){for(Q.c=j(l.j7,H,6,n,15,1),n=0,D=0;D<e.$o(t.k,Q.a);D++)t.c[e._o(t.k,Q.a,D)]||(o=e.Zo(t.k,Q.a,D),Q.c[n++]=t.q[o]<<16|o);for(w.vzb(Q.c,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),r=0;r<Q.c.length;r++)Q.c[r]=n1&Q.c[r];}}for(P=j(l.Feb,ht,6,t.k.g,16,1),t.e=j(l.j7,H,6,t.k.g,15,1),L=new y.Bqb(t.g);L.a<L.c.a.length;)if(Q=y.Aqb(L),Q.c!=null){for(Q.d=j(l.Feb,ht,6,Q.c.length,16,1),r=0;r<Q.c.length;r++)for(u=0;u<e.$o(t.k,Q.a);u++)if(Q.c[r]==e.Zo(t.k,Q.a,u))if(A=e._o(t.k,Q.a,u),t.c[A])P[t.e[A]]=false;else {for(t.c[A]=true,Q.d[r]=true,t.e[A]=1;P[t.e[A]];)++t.e[A];P[t.e[A]]=true;}}},e.mr=function(t){var A,n,P,r;for(P=-1,r=St,A=0;A<t.k.f;A++)t.b[A]||(n=t.a[A],e.Jm(t.k,A)&&(n+=x0),e.$o(t.k,A)==0?n+=1056964608:n+=e.$o(t.k,A)<<24,r>n&&(r=n,P=A));return P;},e.nr=function(t){var A,n,P,r,i;for(t.a=t.d.f,t.b=j(l.Feb,ht,6,t.k.f,16,1),t.c=j(l.Feb,ht,6,t.k.g,16,1),t.g=new y.Mj(),A=e.mr(t);A!=-1;){if(n=t.g.a.length,e.fr(t,new e.$u(A,-1,-1,false,false),n),e.$o(t.k,A)!=0)for(e.dr(t,n,false);n<t.g.a.length-1;){for(;e.tr(t,y.xj(t.g,n).a);)e.dr(t,n,true);++n;}A=e.mr(t);}for(t.q=j(l.j7,H,6,t.k.f,15,1),P=0,i=new y.Bqb(t.g);i.a<i.c.a.length;)r=y.Aqb(i),t.q[r.a]=P++;},e.or=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;if(o=false,e.Qo(t.k,A)!=0&&e.$o(t.k,A)==2&&e.ap(t.k,A,0)==2&&e.ap(t.k,A,1)==2){for(D=0;D<e.$o(t.k,A);D++)if(P=e.Zo(t.k,A,D),P!=n&&e.$o(t.k,P)==2){o=true;break;}for(i=0;i<e.$o(t.k,A);i++){for(P=e.Zo(t.k,A,i),c=0,b=j(l.j7,H,6,3,15,1),L=0;L<e.$o(t.k,P);L++)b[c]=e.Zo(t.k,P,L),b[c]!=A&&++c;c==2&&t.q[b[0]]<t.q[b[1]]^b[0]<b[1]&&(o=!o);}}else {for(s=j(l.j7,H,6,4,15,1),h=j(l.j7,H,6,4,15,1),u=0,n!=-1&&(s[u]=n,h[u++]=8*t.q[n]),(e.op(t.k,A)!=0||e.$o(t.k,A)==3)&&(s[u]=St,h[u++]=8*t.q[A]),i=0;i<e.$o(t.k,A);i++)P=e.Zo(t.k,A,i),P!=n&&(s[u]=P,h[u++]=e.rr(t,A,i));o=e.vr(s,h);}return Q=e.Nl(t.k,A)==1^o,e.ym(t.k,A)&&(r=e.ug(t.d,A),t.n[r]||(t.n[r]=true,t.o[r]=Q),t.o[r]&&(Q=!Q)),Q?"@@":"@";},e.pr=function(t,A,n){var P,r,i,D,u,o,Q,L;switch(u=w.aAb(n.a).length,0<u?n.a=T.Bjb(n.a,0,0):0>u&&(n.a+=T.Hjb(j(l.g7,e0,6,-u,15,1))),o=e.Ol(t.k,A),r=Jt(t0(J(o,mA),3)),r){case 20971520:n.a+="+0";break;case 25165824:e.Bl(t.k,A)==0&&(n.a+="-");break;case 12582912:e.Bl(t.k,A)==0&&(n.a+="+");}return P=J(o,Xn),et(P,2)==0?n.a+=";a":et(P,4)==0&&(n.a+=";A"),i=J(o,X0),et(i,0)!=0&&(et(i,1792)==0?n.a+=";H0":et(i,1664)==0?n.a+=";H1":et(i,1408)==0?n.a+=";H2":et(i,896)==0?n.a+=";H3":et(i,128)==0?n.a+=";!H0":et(i,384)==0?n.a+=";!H0;!H1":et(i,1536)==0?n.a+=";!H2;!H3":et(i,i0)==0&&(n.a+=";!H3")),L=J(o,120),et(L,8)==0?n.a+=";!R0":et(L,16)==0?n.a+=";!R1":et(L,32)==0?n.a+=";!R2":et(L,64)==0?n.a+=";!R3":et(L,112)==0?n.a+=";R0":et(L,104)==0?n.a+=";R1":et(L,88)==0?n.a+=";R2":et(L,56)==0&&(n.a+=";R3"),Q=J(o,EA),et(Q,QA)==0?n.a+=";!r"+y0(Q):et(Q,Z8)==0?n.a+=";r"+y0(Q):et(Q,0)!=0&&(pt(J(Q,De),0)?(H1(J(Q,QA),0)&&(n.a+=";!r0"+y0(Q)),H1(J(Q,SA),0)&&(n.a+=";!r3"+y0(Q)),H1(J(Q,VA),0)&&(n.a+=";!r4"+y0(Q)),H1(J(Q,xA),0)&&(n.a+=";!r5"+y0(Q)),H1(J(Q,zA),0)&&(n.a+=";!r6"+y0(Q)),H1(J(Q,NA),0)&&(n.a+=";!r7"+y0(Q))):(n.a+=";",pt(J(Q,QA),0)&&(n.a+="r0,"+y0(Q)),pt(J(Q,SA),0)&&(n.a+="r3,"+y0(Q)),pt(J(Q,VA),0)&&(n.a+="r4,"+y0(Q)),pt(J(Q,xA),0)&&(n.a+="r5,"+y0(Q)),pt(J(Q,zA),0)&&(n.a+="r6,"+y0(Q)),pt(J(Q,NA),0)&&(n.a+="r7,"+y0(Q)),T.Ggb(n,w.aAb(n.a).length-1))),et(Q,0)==0&&(Q=t0(J(o,kA),22),et(Q,0)!=0&&(n.a+=";r"+y0(Q))),D=J(o,aA),et(D,Pn)==0&&(n.a+=";D1"),et(D,pe)==0&&(n.a+=";D2"),et(D,Re)==0&&(n.a+=";D3"),et(D,rn)==0&&(n.a+=";!D3;!D4"),et(D,v0)==0&&(n.a+=";!D4"),et(D,te)==0&&(n.a+=";!D0;!D1"),et(D,iA)==0&&(n.a+=";!D0;!D1;!D2"),et(D,le)==0&&(n.a+=";!D0;!D1;!D2;!D3"),pt(J(o,j1),0)&&T.Sjb(n,";D"+e.$o(t.k,A)),pt(J(o,Z1),0)&&T.Sjb(n,";!D"+e.$o(t.k,A)),w.aAb(n.a).length==0?null:n.a;},e.qr=function(t){return t.p==null&&(t.p=e.kr(t)),t.p;},e.rr=function(t,A,n){var P,r,i,D,u;if(P=e._o(t.k,A,n),D=e.Zo(t.k,A,n),t.e[P]!=0){for(u=8*t.q[A]+1,r=y.xj(t.g,t.q[A]).c,i=0;i<r.length&&D!=r[i];i++)++u;return u;}return 8*t.q[D];},e.sr=function(t,A){var n,P,r,i,D;for(n=-1,P=-1,i=0;i<e.$o(t.k,A);i++)r=e.Zo(t.k,A,i),D=(e.ap(t.k,A,i)<<24)+t.a[r],!t.b[r]&&(n==-1||P<D)&&(n=i,P=D);return n;},e.tr=function(t,A){var n;for(n=0;n<e.$o(t.k,A);n++)if(!t.b[e.Zo(t.k,A,n)])return  true;return  false;},e.ur=function(t,A,n){var P,r;return P=y.xj(t.g,t.q[A]),P.g==n?false:(r=y.xj(t.g,t.q[n]),r.g==A?true:e.Zu(r,A));},e.vr=function(t,A){var n,P,r;for(P=false,n=1;n<4;n++)for(r=0;r<n;r++)t[r]>t[n]&&(P=!P),A[r]>A[n]&&(P=!P);return P;},e.wr=function(t){return t>=5&&t<=9||t>=15&&t<=17||t==35||t==53;},e.xr=function(t){e.yr.call(this,t,0);},e.yr=function(t,A){this.k=t,this.j=A;},e.zr=function(t){var A,n,P,r;for(r=new T.Ujb(),n=0;n<t.g.a.length;n++)n!=0&&(r.a+="."),T.Sjb(r,e.qr(new e.yr(y.xj(t.g,n),2)));for(r.a+=">",P=0;P<t.a.a.length;P++)P!=0&&(r.a+="."),T.Sjb(r,e.qr(new e.xr(y.xj(t.a,P))));for(r.a+=">",A=0;A<t.f.a.length;A++)A!=0&&(r.a+="."),T.Sjb(r,e.qr(new e.yr(y.xj(t.f,A),2)));return r.a;},N(111,1,{},e.xr,e.yr),U.j=0,l.N7=O(111),e.Ar=function(t,A){this.b=t,this.a=A;},N(3,1,{3:1},e.Ar),U.a=0,U.b=0,l.O7=O(3),e.Cr=function(){e.Cr=D1,e.Br=v(F(l.O7,2),c0,17,0,[null,v(F(l.O7,1),Vt,3,0,[new e.Ar(0,1.007825032),new e.Ar(1,2.014101778),new e.Ar(2,3.016049268),new e.Ar(3,4.027834627),new e.Ar(4,5.039542911),new e.Ar(5,6.044942608)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(1,3.01602931),new e.Ar(2,4.00260325),new e.Ar(3,5.012223628),new e.Ar(4,6.018888072),new e.Ar(5,7.028030527),new e.Ar(6,8.033921838),new e.Ar(7,9.043820323),new e.Ar(8,10.052399713)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(1,4.027182329),new e.Ar(2,5.012537796),new e.Ar(3,6.015122281),new e.Ar(4,7.016004049),new e.Ar(5,8.02248667),new e.Ar(6,9.026789122),new e.Ar(7,10.035480884),new e.Ar(8,11.043796166),new e.Ar(9,12.05378)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(1,5.04079),new e.Ar(2,6.019725804),new e.Ar(3,7.016929246),new e.Ar(4,8.005305094),new e.Ar(5,9.012182135),new e.Ar(6,10.01353372),new e.Ar(7,11.021657653),new e.Ar(8,12.026920631),new e.Ar(9,13.036133834),new e.Ar(10,14.042815522)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(2,7.029917389),new e.Ar(3,8.024606713),new e.Ar(4,9.013328806),new e.Ar(5,10.012937027),new e.Ar(6,11.009305466),new e.Ar(7,12.014352109),new e.Ar(8,13.017780267),new e.Ar(9,14.025404064),new e.Ar(10,15.031097291),new e.Ar(11,16.039808836),new e.Ar(12,17.046931399),new e.Ar(13,18.05617),new e.Ar(14,19.06373)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(2,8.037675026),new e.Ar(3,9.031040087),new e.Ar(4,10.01685311),new e.Ar(5,11.011433818),new e.Ar(6,12),new e.Ar(7,13.003354838),new e.Ar(8,14.003241988),new e.Ar(9,15.010599258),new e.Ar(10,16.014701243),new e.Ar(11,17.022583712),new e.Ar(12,18.026757058),new e.Ar(13,19.035248094),new e.Ar(14,20.040322395),new e.Ar(15,21.04934),new e.Ar(16,22.05645)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(3,10.042618),new e.Ar(4,11.026796226),new e.Ar(5,12.018613202),new e.Ar(6,13.005738584),new e.Ar(7,14.003074005),new e.Ar(8,15.000108898),new e.Ar(9,16.006101417),new e.Ar(10,17.008449673),new e.Ar(11,18.014081827),new e.Ar(12,19.017026896),new e.Ar(13,20.023367295),new e.Ar(14,21.027087574),new e.Ar(15,22.034440259),new e.Ar(16,23.04051),new e.Ar(17,24.0505)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(4,12.034404776),new e.Ar(5,13.0248104),new e.Ar(6,14.008595285),new e.Ar(7,15.003065386),new e.Ar(8,15.994914622),new e.Ar(9,16.999131501),new e.Ar(10,17.999160419),new e.Ar(11,19.00357873),new e.Ar(12,20.00407615),new e.Ar(13,21.008654631),new e.Ar(14,22.009967157),new e.Ar(15,23.015691325),new e.Ar(16,24.020369922),new e.Ar(17,25.02914),new e.Ar(18,26.03775)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(5,14.03608),new e.Ar(6,15.018010856),new e.Ar(7,16.01146573),new e.Ar(8,17.002095238),new e.Ar(9,18.000937667),new e.Ar(10,18.998403205),new e.Ar(11,19.999981324),new e.Ar(12,20.999948921),new e.Ar(13,22.00299925),new e.Ar(14,23.003574385),new e.Ar(15,24.008099371),new e.Ar(16,25.012094963),new e.Ar(17,26.019633157),new e.Ar(18,27.026892316),new e.Ar(19,28.03567),new e.Ar(20,29.04326)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(6,16.025756907),new e.Ar(7,17.017697565),new e.Ar(8,18.005697066),new e.Ar(9,19.001879839),new e.Ar(10,19.992440176),new e.Ar(11,20.993846744),new e.Ar(12,21.99138551),new e.Ar(13,22.994467337),new e.Ar(14,23.993615074),new e.Ar(15,24.997789899),new e.Ar(16,26.000461498),new e.Ar(17,27.0076152),new e.Ar(18,28.012108072),new e.Ar(19,29.019345902),new e.Ar(20,30.023872),new e.Ar(21,31.03311),new e.Ar(22,32.03991)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(7,18.02718),new e.Ar(8,19.01387945),new e.Ar(9,20.00734826),new e.Ar(10,20.997655099),new e.Ar(11,21.994436782),new e.Ar(12,22.989769675),new e.Ar(13,23.990963332),new e.Ar(14,24.989954352),new e.Ar(15,25.992589898),new e.Ar(16,26.994008702),new e.Ar(17,27.99889041),new e.Ar(18,29.002811301),new e.Ar(19,30.009226487),new e.Ar(20,31.013595108),new e.Ar(21,32.019649792),new e.Ar(22,33.027386),new e.Ar(23,34.0349),new e.Ar(24,35.04418)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(8,20.018862744),new e.Ar(9,21.011714174),new e.Ar(10,21.999574055),new e.Ar(11,22.99412485),new e.Ar(12,23.985041898),new e.Ar(13,24.985837023),new e.Ar(14,25.98259304),new e.Ar(15,26.984340742),new e.Ar(16,27.983876703),new e.Ar(17,28.988554743),new e.Ar(18,29.990464529),new e.Ar(19,30.996548459),new e.Ar(20,31.999145889),new e.Ar(21,33.005586975),new e.Ar(22,34.00907244),new e.Ar(23,35.018669),new e.Ar(24,36.02245),new e.Ar(25,37.03124)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(8,21.02804),new e.Ar(9,22.01952),new e.Ar(10,23.0072649),new e.Ar(11,23.999940911),new e.Ar(12,24.990428555),new e.Ar(13,25.986891659),new e.Ar(14,26.981538441),new e.Ar(15,27.981910184),new e.Ar(16,28.980444848),new e.Ar(17,29.982960304),new e.Ar(18,30.983946023),new e.Ar(19,31.988124379),new e.Ar(20,32.990869587),new e.Ar(21,33.996927255),new e.Ar(22,34.99993765),new e.Ar(23,36.006351501),new e.Ar(24,37.01031),new e.Ar(25,38.0169),new e.Ar(26,39.0219)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(8,22.03453),new e.Ar(9,23.02552),new e.Ar(10,24.011545711),new e.Ar(11,25.00410664),new e.Ar(12,25.992329935),new e.Ar(13,26.986704764),new e.Ar(14,27.976926533),new e.Ar(15,28.976494719),new e.Ar(16,29.973770218),new e.Ar(17,30.975363275),new e.Ar(18,31.974148129),new e.Ar(19,32.97800052),new e.Ar(20,33.978575745),new e.Ar(21,34.984584158),new e.Ar(22,35.986687363),new e.Ar(23,36.99299599),new e.Ar(24,37.99598),new e.Ar(25,39.0023),new e.Ar(26,40.0058),new e.Ar(27,41.0127),new e.Ar(28,42.0161)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(9,24.03435),new e.Ar(10,25.02026),new e.Ar(11,26.01178),new e.Ar(12,26.999191645),new e.Ar(13,27.99231233),new e.Ar(14,28.981801376),new e.Ar(15,29.978313807),new e.Ar(16,30.973761512),new e.Ar(17,31.973907163),new e.Ar(18,32.971725281),new e.Ar(19,33.973636381),new e.Ar(20,34.973314249),new e.Ar(21,35.978259824),new e.Ar(22,36.979608338),new e.Ar(23,37.98447),new e.Ar(24,38.98642),new e.Ar(25,39.99105),new e.Ar(26,40.9948),new e.Ar(27,42.00009),new e.Ar(28,43.00331),new e.Ar(29,44.00988),new e.Ar(30,45.01514),new e.Ar(31,46.02383)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(10,26.02788),new e.Ar(11,27.018795),new e.Ar(12,28.004372661),new e.Ar(13,28.996608805),new e.Ar(14,29.984902954),new e.Ar(15,30.979554421),new e.Ar(16,31.97207069),new e.Ar(17,32.971458497),new e.Ar(18,33.967866831),new e.Ar(19,34.96903214),new e.Ar(20,35.96708088),new e.Ar(21,36.971125716),new e.Ar(22,37.971163443),new e.Ar(23,38.975135275),new e.Ar(24,39.97547),new e.Ar(25,40.98003),new e.Ar(26,41.98149),new e.Ar(27,42.9866),new e.Ar(28,43.98832),new e.Ar(29,44.99482),new e.Ar(30,45.99957),new e.Ar(31,47.00762),new e.Ar(32,48.01299),new e.Ar(33,49.02201)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(11,28.02851),new e.Ar(12,29.01411),new e.Ar(13,30.00477),new e.Ar(14,30.992416014),new e.Ar(15,31.985688908),new e.Ar(16,32.977451798),new e.Ar(17,33.973761967),new e.Ar(18,34.968852707),new e.Ar(19,35.968306945),new e.Ar(20,36.9659026),new e.Ar(21,37.96801055),new e.Ar(22,38.968007677),new e.Ar(23,39.970415555),new e.Ar(24,40.970650212),new e.Ar(25,41.973174994),new e.Ar(26,42.974203385),new e.Ar(27,43.978538712),new e.Ar(28,44.9797),new e.Ar(29,45.98412),new e.Ar(30,46.98795),new e.Ar(31,47.99485),new e.Ar(32,48.99989),new e.Ar(33,50.00773),new e.Ar(34,51.01353)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(12,30.02156),new e.Ar(13,31.012126),new e.Ar(14,31.99766066),new e.Ar(15,32.989928719),new e.Ar(16,33.980270118),new e.Ar(17,34.975256726),new e.Ar(18,35.967546282),new e.Ar(19,36.966775912),new e.Ar(20,37.962732161),new e.Ar(21,38.964313413),new e.Ar(22,39.962383123),new e.Ar(23,40.964500828),new e.Ar(24,41.963046386),new e.Ar(25,42.965670701),new e.Ar(26,43.965365269),new e.Ar(27,44.968094979),new e.Ar(28,45.968093467),new e.Ar(29,46.972186238),new e.Ar(30,47.97507),new e.Ar(31,48.98218),new e.Ar(32,49.98594),new e.Ar(33,50.99324),new e.Ar(34,51.99817),new e.Ar(35,53.006227)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(13,32.02192),new e.Ar(14,33.00726),new e.Ar(15,33.99841),new e.Ar(16,34.988011615),new e.Ar(17,35.981293405),new e.Ar(18,36.973376915),new e.Ar(19,37.969080107),new e.Ar(20,38.963706861),new e.Ar(21,39.963998672),new e.Ar(22,40.961825972),new e.Ar(23,41.962403059),new e.Ar(24,42.960715746),new e.Ar(25,43.961556146),new e.Ar(26,44.960699658),new e.Ar(27,45.961976203),new e.Ar(28,46.961677807),new e.Ar(29,47.965512946),new e.Ar(30,48.967450084),new e.Ar(31,49.972782832),new e.Ar(32,50.97638),new e.Ar(33,51.98261),new e.Ar(34,52.98712),new e.Ar(35,53.99399),new e.Ar(36,54.999388)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(14,34.01412),new e.Ar(15,35.004765),new e.Ar(16,35.993087234),new e.Ar(17,36.985871505),new e.Ar(18,37.976318637),new e.Ar(19,38.970717729),new e.Ar(20,39.962591155),new e.Ar(21,40.962278349),new e.Ar(22,41.958618337),new e.Ar(23,42.958766833),new e.Ar(24,43.955481094),new e.Ar(25,44.956185938),new e.Ar(26,45.953692759),new e.Ar(27,46.954546459),new e.Ar(28,47.952533512),new e.Ar(29,48.955673302),new e.Ar(30,49.957518286),new e.Ar(31,50.961474238),new e.Ar(32,51.9651),new e.Ar(33,52.97005),new e.Ar(34,53.97468),new e.Ar(35,54.98055),new e.Ar(36,55.98579),new e.Ar(37,56.992356)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(15,36.01492),new e.Ar(16,37.00305),new e.Ar(17,37.9947),new e.Ar(18,38.984790009),new e.Ar(19,39.977964014),new e.Ar(20,40.969251316),new e.Ar(21,41.965516761),new e.Ar(22,42.96115098),new e.Ar(23,43.959403048),new e.Ar(24,44.955910243),new e.Ar(25,45.95517025),new e.Ar(26,46.952408027),new e.Ar(27,47.952234991),new e.Ar(28,48.950024065),new e.Ar(29,49.952187008),new e.Ar(30,50.9536027),new e.Ar(31,51.95665),new e.Ar(32,52.95817),new e.Ar(33,53.963),new e.Ar(34,54.9694),new e.Ar(35,55.97266),new e.Ar(36,56.97704),new e.Ar(37,57.98307),new e.Ar(38,58.988041)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(16,38.00977),new e.Ar(17,39.001323),new e.Ar(18,39.990498907),new e.Ar(19,40.983131),new e.Ar(20,41.973031622),new e.Ar(21,42.968523342),new e.Ar(22,43.959690235),new e.Ar(23,44.958124349),new e.Ar(24,45.952629491),new e.Ar(25,46.951763792),new e.Ar(26,47.947947053),new e.Ar(27,48.947870789),new e.Ar(28,49.944792069),new e.Ar(29,50.946616017),new e.Ar(30,51.946898175),new e.Ar(31,52.949731709),new e.Ar(32,53.95087),new e.Ar(33,54.95512),new e.Ar(34,55.95799),new e.Ar(35,56.9643),new e.Ar(36,57.96611),new e.Ar(37,58.97196),new e.Ar(38,59.97564),new e.Ar(39,60.982018)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(17,40.01109),new e.Ar(18,40.99974),new e.Ar(19,41.99123),new e.Ar(20,42.98065),new e.Ar(21,43.9744),new e.Ar(22,44.965782286),new e.Ar(23,45.960199491),new e.Ar(24,46.954906918),new e.Ar(25,47.95225448),new e.Ar(26,48.948516914),new e.Ar(27,49.947162792),new e.Ar(28,50.943963675),new e.Ar(29,51.944779658),new e.Ar(30,52.944342517),new e.Ar(31,53.946444381),new e.Ar(32,54.947238194),new e.Ar(33,55.95036),new e.Ar(34,56.95236),new e.Ar(35,57.95665),new e.Ar(36,58.9593),new e.Ar(37,59.9645),new e.Ar(38,60.96741),new e.Ar(39,61.97314),new e.Ar(40,62.97675)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(18,42.00643),new e.Ar(19,42.997707),new e.Ar(20,43.98547),new e.Ar(21,44.97916),new e.Ar(22,45.968361649),new e.Ar(23,46.962906512),new e.Ar(24,47.954035861),new e.Ar(25,48.951341135),new e.Ar(26,49.946049607),new e.Ar(27,50.944771767),new e.Ar(28,51.940511904),new e.Ar(29,52.940653781),new e.Ar(30,53.938884921),new e.Ar(31,54.940844164),new e.Ar(32,55.940645238),new e.Ar(33,56.9437538),new e.Ar(34,57.94425),new e.Ar(35,58.94863),new e.Ar(36,59.94973),new e.Ar(37,60.95409),new e.Ar(38,61.9558),new e.Ar(39,62.96186),new e.Ar(40,63.9642),new e.Ar(41,64.97037)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(19,44.00687),new e.Ar(20,44.99451),new e.Ar(21,45.98672),new e.Ar(22,46.9761),new e.Ar(23,47.96887),new e.Ar(24,48.959623415),new e.Ar(25,49.95424396),new e.Ar(26,50.948215487),new e.Ar(27,51.945570079),new e.Ar(28,52.941294702),new e.Ar(29,53.940363247),new e.Ar(30,54.938049636),new e.Ar(31,55.938909366),new e.Ar(32,56.938287458),new e.Ar(33,57.939986451),new e.Ar(34,58.940447166),new e.Ar(35,59.943193998),new e.Ar(36,60.94446),new e.Ar(37,61.94797),new e.Ar(38,62.94981),new e.Ar(39,63.95373),new e.Ar(40,64.9561),new e.Ar(41,65.96082),new e.Ar(42,66.96382)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(19,45.01456),new e.Ar(20,46.00081),new e.Ar(21,46.99289),new e.Ar(22,47.98056),new e.Ar(23,48.97361),new e.Ar(24,49.962993316),new e.Ar(25,50.956824936),new e.Ar(26,51.948116526),new e.Ar(27,52.945312282),new e.Ar(28,53.939614836),new e.Ar(29,54.938298029),new e.Ar(30,55.934942133),new e.Ar(31,56.935398707),new e.Ar(32,57.933280458),new e.Ar(33,58.934880493),new e.Ar(34,59.934076943),new e.Ar(35,60.936749461),new e.Ar(36,61.936770495),new e.Ar(37,62.940118442),new e.Ar(38,63.94087),new e.Ar(39,64.94494),new e.Ar(40,65.94598),new e.Ar(41,66.95),new e.Ar(42,67.95251),new e.Ar(43,68.9577)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(21,48.00176),new e.Ar(22,48.98972),new e.Ar(23,49.98154),new e.Ar(24,50.97072),new e.Ar(25,51.96359),new e.Ar(26,52.954224985),new e.Ar(27,53.948464147),new e.Ar(28,54.942003149),new e.Ar(29,55.939843937),new e.Ar(30,56.936296235),new e.Ar(31,57.935757571),new e.Ar(32,58.933200194),new e.Ar(33,59.933822196),new e.Ar(34,60.932479381),new e.Ar(35,61.934054212),new e.Ar(36,62.933615218),new e.Ar(37,63.935813523),new e.Ar(38,64.936484581),new e.Ar(39,65.939825412),new e.Ar(40,66.94061),new e.Ar(41,67.94436),new e.Ar(42,68.9452),new e.Ar(43,69.94981),new e.Ar(44,70.95173),new e.Ar(45,71.95641)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(22,49.99593),new e.Ar(23,50.98772),new e.Ar(24,51.97568),new e.Ar(25,52.96846),new e.Ar(26,53.957910508),new e.Ar(27,54.951336329),new e.Ar(28,55.942136339),new e.Ar(29,56.939800489),new e.Ar(30,57.935347922),new e.Ar(31,58.934351553),new e.Ar(32,59.930790633),new e.Ar(33,60.931060442),new e.Ar(34,61.928348763),new e.Ar(35,62.929672948),new e.Ar(36,63.927969574),new e.Ar(37,64.930088013),new e.Ar(38,65.929115232),new e.Ar(39,66.931569638),new e.Ar(40,67.931844932),new e.Ar(41,68.935181837),new e.Ar(42,69.93614),new e.Ar(43,70.94),new e.Ar(44,71.9413),new e.Ar(45,72.94608),new e.Ar(46,73.94791),new e.Ar(47,74.95297),new e.Ar(48,75.95533),new e.Ar(49,76.96083),new e.Ar(50,77.9638)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(23,51.99718),new e.Ar(24,52.98555),new e.Ar(25,53.97671),new e.Ar(26,54.96605),new e.Ar(27,55.95856),new e.Ar(28,56.949215695),new e.Ar(29,57.944540734),new e.Ar(30,58.939504114),new e.Ar(31,59.937368123),new e.Ar(32,60.933462181),new e.Ar(33,61.932587299),new e.Ar(34,62.929601079),new e.Ar(35,63.929767865),new e.Ar(36,64.927793707),new e.Ar(37,65.928873041),new e.Ar(38,66.927750294),new e.Ar(39,67.929637875),new e.Ar(40,68.929425281),new e.Ar(41,69.932409287),new e.Ar(42,70.932619818),new e.Ar(43,71.93552),new e.Ar(44,72.93649),new e.Ar(45,73.9402),new e.Ar(46,74.9417),new e.Ar(47,75.94599),new e.Ar(48,76.94795),new e.Ar(49,77.95281),new e.Ar(50,78.95528),new e.Ar(51,79.96189)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(24,53.99295),new e.Ar(25,54.98398),new e.Ar(26,55.97238),new e.Ar(27,56.96491),new e.Ar(28,57.954596465),new e.Ar(29,58.949267074),new e.Ar(30,59.941832031),new e.Ar(31,60.939513907),new e.Ar(32,61.934334132),new e.Ar(33,62.933215563),new e.Ar(34,63.929146578),new e.Ar(35,64.929245079),new e.Ar(36,65.926036763),new e.Ar(37,66.927130859),new e.Ar(38,67.924847566),new e.Ar(39,68.926553538),new e.Ar(40,69.92532487),new e.Ar(41,70.927727195),new e.Ar(42,71.926861122),new e.Ar(43,72.929779469),new e.Ar(44,73.929458261),new e.Ar(45,74.932937379),new e.Ar(46,75.933394207),new e.Ar(47,76.937085857),new e.Ar(48,77.938569576),new e.Ar(49,78.942095175),new e.Ar(50,79.944414722),new e.Ar(51,80.95048),new e.Ar(52,81.95484)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(25,55.99491),new e.Ar(26,56.98293),new e.Ar(27,57.97425),new e.Ar(28,58.96337),new e.Ar(29,59.95706),new e.Ar(30,60.94917),new e.Ar(31,61.944179608),new e.Ar(32,62.939141527),new e.Ar(33,63.936838307),new e.Ar(34,64.932739322),new e.Ar(35,65.931592355),new e.Ar(36,66.928204915),new e.Ar(37,67.927983497),new e.Ar(38,68.925580912),new e.Ar(39,69.926027741),new e.Ar(40,70.92470501),new e.Ar(41,71.92636935),new e.Ar(42,72.925169832),new e.Ar(43,73.926940999),new e.Ar(44,74.926500645),new e.Ar(45,75.928928262),new e.Ar(46,76.929281189),new e.Ar(47,77.93165595),new e.Ar(48,78.932916371),new e.Ar(49,79.936588154),new e.Ar(50,80.937752955),new e.Ar(51,81.94316),new e.Ar(52,82.94687),new e.Ar(53,83.95234)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(26,57.99101),new e.Ar(27,58.98175),new e.Ar(28,59.97019),new e.Ar(29,60.96379),new e.Ar(30,61.95465),new e.Ar(31,62.94964),new e.Ar(32,63.941572638),new e.Ar(33,64.939440762),new e.Ar(34,65.933846798),new e.Ar(35,66.932738415),new e.Ar(36,67.928097266),new e.Ar(37,68.927972002),new e.Ar(38,69.924250365),new e.Ar(39,70.924953991),new e.Ar(40,71.922076184),new e.Ar(41,72.923459361),new e.Ar(42,73.921178213),new e.Ar(43,74.922859494),new e.Ar(44,75.921402716),new e.Ar(45,76.923548462),new e.Ar(46,77.922852886),new e.Ar(47,78.92540156),new e.Ar(48,79.925444764),new e.Ar(49,80.928821065),new e.Ar(50,81.929550326),new e.Ar(51,82.93451),new e.Ar(52,83.93731),new e.Ar(53,84.94269),new e.Ar(54,85.94627)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(27,59.99313),new e.Ar(28,60.98062),new e.Ar(29,61.9732),new e.Ar(30,62.96369),new e.Ar(31,63.957572),new e.Ar(32,64.949484),new e.Ar(33,65.944099147),new e.Ar(34,66.939190417),new e.Ar(35,67.936792976),new e.Ar(36,68.932280154),new e.Ar(37,69.930927811),new e.Ar(38,70.927114724),new e.Ar(39,71.926752647),new e.Ar(40,72.923825288),new e.Ar(41,73.923929076),new e.Ar(42,74.921596417),new e.Ar(43,75.922393933),new e.Ar(44,76.920647703),new e.Ar(45,77.921828577),new e.Ar(46,78.920948498),new e.Ar(47,79.922578162),new e.Ar(48,80.922132884),new e.Ar(49,81.924504668),new e.Ar(50,82.924980625),new e.Ar(51,83.92906),new e.Ar(52,84.93181),new e.Ar(53,85.93623),new e.Ar(54,86.93958),new e.Ar(55,87.94456),new e.Ar(56,88.94923)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(31,64.96466),new e.Ar(32,65.95521),new e.Ar(33,66.95009),new e.Ar(34,67.94187),new e.Ar(35,68.939562155),new e.Ar(36,69.933504),new e.Ar(37,70.931868378),new e.Ar(38,71.927112313),new e.Ar(39,72.9267668),new e.Ar(40,73.922476561),new e.Ar(41,74.922523571),new e.Ar(42,75.919214107),new e.Ar(43,76.91991461),new e.Ar(44,77.917309522),new e.Ar(45,78.918499802),new e.Ar(46,79.916521828),new e.Ar(47,80.917992931),new e.Ar(48,81.9167),new e.Ar(49,82.919119072),new e.Ar(50,83.918464523),new e.Ar(51,84.922244678),new e.Ar(52,85.924271165),new e.Ar(53,86.928520749),new e.Ar(54,87.931423982),new e.Ar(55,88.93602),new e.Ar(56,89.93942),new e.Ar(57,90.94537),new e.Ar(58,91.94933)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(32,66.96479),new e.Ar(33,67.958248),new e.Ar(34,68.950178),new e.Ar(35,69.944208),new e.Ar(36,70.939246),new e.Ar(37,71.936496876),new e.Ar(38,72.931794889),new e.Ar(39,73.929891152),new e.Ar(40,74.92577641),new e.Ar(41,75.924541974),new e.Ar(42,76.921380123),new e.Ar(43,77.92114613),new e.Ar(44,78.918337647),new e.Ar(45,79.918529952),new e.Ar(46,80.91629106),new e.Ar(47,81.916804666),new e.Ar(48,82.915180219),new e.Ar(49,83.916503685),new e.Ar(50,84.915608027),new e.Ar(51,85.918797162),new e.Ar(52,86.920710713),new e.Ar(53,87.924065908),new e.Ar(54,88.92638726),new e.Ar(55,89.930634988),new e.Ar(56,90.9339653),new e.Ar(57,91.939255258),new e.Ar(58,92.9431),new e.Ar(59,93.94868)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(33,68.96532),new e.Ar(34,69.95601),new e.Ar(35,70.95051),new e.Ar(36,71.94190754),new e.Ar(37,72.938931115),new e.Ar(38,73.933258225),new e.Ar(39,74.931033794),new e.Ar(40,75.925948304),new e.Ar(41,76.92466788),new e.Ar(42,77.920386271),new e.Ar(43,78.920082992),new e.Ar(44,79.91637804),new e.Ar(45,80.916592419),new e.Ar(46,81.913484601),new e.Ar(47,82.914135952),new e.Ar(48,83.911506627),new e.Ar(49,84.912526954),new e.Ar(50,85.910610313),new e.Ar(51,86.913354251),new e.Ar(52,87.914446951),new e.Ar(53,88.917632505),new e.Ar(54,89.919523803),new e.Ar(55,90.923442418),new e.Ar(56,91.926152752),new e.Ar(57,92.931265246),new e.Ar(58,93.934362),new e.Ar(59,94.93984),new e.Ar(60,95.94307),new e.Ar(61,96.94856)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(34,70.96532),new e.Ar(35,71.95908),new e.Ar(36,72.950366),new e.Ar(37,73.944470376),new e.Ar(38,74.938569199),new e.Ar(39,75.935071448),new e.Ar(40,76.930406599),new e.Ar(41,77.928141485),new e.Ar(42,78.923996719),new e.Ar(43,79.922519322),new e.Ar(44,80.918994165),new e.Ar(45,81.918207691),new e.Ar(46,82.915111951),new e.Ar(47,83.914384676),new e.Ar(48,84.911789341),new e.Ar(49,85.91116708),new e.Ar(50,86.909183465),new e.Ar(51,87.911318556),new e.Ar(52,88.912279939),new e.Ar(53,89.914808941),new e.Ar(54,90.91653416),new e.Ar(55,91.919725442),new e.Ar(56,92.922032765),new e.Ar(57,93.926407326),new e.Ar(58,94.92931926),new e.Ar(59,95.934283962),new e.Ar(60,96.937342863),new e.Ar(61,97.941703557),new e.Ar(62,98.945420616),new e.Ar(63,99.94987),new e.Ar(64,100.953195994),new e.Ar(65,101.95921)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(35,72.96597),new e.Ar(36,73.95631),new e.Ar(37,74.94992),new e.Ar(38,75.94161),new e.Ar(39,76.937761511),new e.Ar(40,77.932179362),new e.Ar(41,78.929707076),new e.Ar(42,79.924524588),new e.Ar(43,80.923213095),new e.Ar(44,81.918401258),new e.Ar(45,82.917555029),new e.Ar(46,83.913424778),new e.Ar(47,84.912932689),new e.Ar(48,85.909262351),new e.Ar(49,86.908879316),new e.Ar(50,87.905614339),new e.Ar(51,88.907452906),new e.Ar(52,89.907737596),new e.Ar(53,90.910209845),new e.Ar(54,91.911029895),new e.Ar(55,92.91402241),new e.Ar(56,93.915359856),new e.Ar(57,94.919358213),new e.Ar(58,95.921680473),new e.Ar(59,96.926148757),new e.Ar(60,97.928471177),new e.Ar(61,98.933315038),new e.Ar(62,99.935351729),new e.Ar(63,100.940517434),new e.Ar(64,101.943018795),new e.Ar(65,102.94895),new e.Ar(66,103.95233)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(38,76.94962),new e.Ar(39,77.9435),new e.Ar(40,78.937350712),new e.Ar(41,79.931982402),new e.Ar(42,80.929128719),new e.Ar(43,81.926792071),new e.Ar(44,82.922352572),new e.Ar(45,83.920387768),new e.Ar(46,84.916427076),new e.Ar(47,85.914887724),new e.Ar(48,86.910877833),new e.Ar(49,87.909503361),new e.Ar(50,88.905847902),new e.Ar(51,89.907151443),new e.Ar(52,90.907303415),new e.Ar(53,91.908946832),new e.Ar(54,92.909581582),new e.Ar(55,93.911594008),new e.Ar(56,94.912823709),new e.Ar(57,95.915897787),new e.Ar(58,96.918131017),new e.Ar(59,97.922219525),new e.Ar(60,98.924634736),new e.Ar(61,99.927756402),new e.Ar(62,100.930313395),new e.Ar(63,101.933555501),new e.Ar(64,102.93694),new e.Ar(65,103.94145),new e.Ar(66,104.94509),new e.Ar(67,105.95022)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(39,78.94916),new e.Ar(40,79.94055),new e.Ar(41,80.936815296),new e.Ar(42,81.931086249),new e.Ar(43,82.92865213),new e.Ar(44,83.92325),new e.Ar(45,84.92146522),new e.Ar(46,85.916472851),new e.Ar(47,86.914816578),new e.Ar(48,87.910226179),new e.Ar(49,88.908888916),new e.Ar(50,89.904703679),new e.Ar(51,90.905644968),new e.Ar(52,91.905040106),new e.Ar(53,92.906475627),new e.Ar(54,93.906315765),new e.Ar(55,94.908042739),new e.Ar(56,95.908275675),new e.Ar(57,96.910950716),new e.Ar(58,97.912746366),new e.Ar(59,98.916511084),new e.Ar(60,99.917761704),new e.Ar(61,100.921139958),new e.Ar(62,101.922981089),new e.Ar(63,102.926597062),new e.Ar(64,103.92878),new e.Ar(65,104.93305),new e.Ar(66,105.93591),new e.Ar(67,106.94086),new e.Ar(68,107.94428)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(40,80.94905),new e.Ar(41,81.94313),new e.Ar(42,82.936703713),new e.Ar(43,83.93357),new e.Ar(44,84.927906486),new e.Ar(45,85.925037588),new e.Ar(46,86.920361435),new e.Ar(47,87.91833144),new e.Ar(48,88.913495503),new e.Ar(49,89.911264109),new e.Ar(50,90.906990538),new e.Ar(51,91.907193214),new e.Ar(52,92.906377543),new e.Ar(53,93.907283457),new e.Ar(54,94.906835178),new e.Ar(55,95.908100076),new e.Ar(56,96.908097144),new e.Ar(57,97.91033069),new e.Ar(58,98.911617864),new e.Ar(59,99.914181434),new e.Ar(60,100.915251567),new e.Ar(61,101.918037417),new e.Ar(62,102.919141297),new e.Ar(63,103.922459464),new e.Ar(64,104.923934023),new e.Ar(65,105.92819),new e.Ar(66,106.93031),new e.Ar(67,107.93501),new e.Ar(68,108.93763),new e.Ar(69,109.94268)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(41,82.94874),new e.Ar(42,83.94009),new e.Ar(43,84.93659),new e.Ar(44,85.930695167),new e.Ar(45,86.92732683),new e.Ar(46,87.921952728),new e.Ar(47,88.919480562),new e.Ar(48,89.913936161),new e.Ar(49,90.911750754),new e.Ar(50,91.90681048),new e.Ar(51,92.906812213),new e.Ar(52,93.905087578),new e.Ar(53,94.905841487),new e.Ar(54,95.904678904),new e.Ar(55,96.906021033),new e.Ar(56,97.905407846),new e.Ar(57,98.907711598),new e.Ar(58,99.907477149),new e.Ar(59,100.910346543),new e.Ar(60,101.910297162),new e.Ar(61,102.913204596),new e.Ar(62,103.913758387),new e.Ar(63,104.916972087),new e.Ar(64,105.918134284),new e.Ar(65,106.921694724),new e.Ar(66,107.923973837),new e.Ar(67,108.92781),new e.Ar(68,109.92973),new e.Ar(69,110.93451),new e.Ar(70,111.93684),new e.Ar(71,112.94203)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(42,84.94894),new e.Ar(43,85.94288),new e.Ar(44,86.93653),new e.Ar(45,87.93283),new e.Ar(46,88.92754288),new e.Ar(47,89.92355583),new e.Ar(48,90.9184282),new e.Ar(49,91.915259655),new e.Ar(50,92.910248473),new e.Ar(51,93.909656309),new e.Ar(52,94.907656454),new e.Ar(53,95.907870803),new e.Ar(54,96.906364843),new e.Ar(55,97.907215692),new e.Ar(56,98.906254554),new e.Ar(57,99.907657594),new e.Ar(58,100.90731438),new e.Ar(59,101.909212938),new e.Ar(60,102.909178805),new e.Ar(61,103.911444898),new e.Ar(62,104.911658043),new e.Ar(63,105.914355408),new e.Ar(64,106.915081691),new e.Ar(65,107.918479973),new e.Ar(66,108.919980998),new e.Ar(67,109.92339),new e.Ar(68,110.92505),new e.Ar(69,111.92924),new e.Ar(70,112.93133),new e.Ar(71,113.93588),new e.Ar(72,114.93828)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(43,86.94918),new e.Ar(44,87.94042),new e.Ar(45,88.93611),new e.Ar(46,89.92978),new e.Ar(47,90.926377434),new e.Ar(48,91.92012),new e.Ar(49,92.917051523),new e.Ar(50,93.911359569),new e.Ar(51,94.910412729),new e.Ar(52,95.907597681),new e.Ar(53,96.907554546),new e.Ar(54,97.905287111),new e.Ar(55,98.905939307),new e.Ar(56,99.904219664),new e.Ar(57,100.905582219),new e.Ar(58,101.904349503),new e.Ar(59,102.906323677),new e.Ar(60,103.905430145),new e.Ar(61,104.907750341),new e.Ar(62,105.907326913),new e.Ar(63,106.909907207),new e.Ar(64,107.910192211),new e.Ar(65,108.913201565),new e.Ar(66,109.913966185),new e.Ar(67,110.91756),new e.Ar(68,111.918821673),new e.Ar(69,112.92254),new e.Ar(70,113.923891981),new e.Ar(71,114.92831),new e.Ar(72,115.93016),new e.Ar(73,116.93479),new e.Ar(74,117.93703)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(44,88.94938),new e.Ar(45,89.94287),new e.Ar(46,90.93655),new e.Ar(47,91.93198),new e.Ar(48,92.92574),new e.Ar(49,93.921698),new e.Ar(50,94.915898541),new e.Ar(51,95.914518212),new e.Ar(52,96.911336643),new e.Ar(53,97.910716431),new e.Ar(54,98.908132101),new e.Ar(55,99.90811663),new e.Ar(56,100.906163526),new e.Ar(57,101.906842845),new e.Ar(58,102.905504182),new e.Ar(59,103.906655315),new e.Ar(60,104.905692444),new e.Ar(61,105.907284615),new e.Ar(62,106.90675054),new e.Ar(63,107.908730768),new e.Ar(64,108.908735621),new e.Ar(65,109.910949525),new e.Ar(66,110.91166),new e.Ar(67,111.913969253),new e.Ar(68,112.91542),new e.Ar(69,113.91734336),new e.Ar(70,114.920124676),new e.Ar(71,115.922746643),new e.Ar(72,116.92535),new e.Ar(73,117.92943),new e.Ar(74,118.93136),new e.Ar(75,119.93578),new e.Ar(76,120.93808)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(45,90.94948),new e.Ar(46,91.94042),new e.Ar(47,92.93591),new e.Ar(48,93.92877),new e.Ar(49,94.92469),new e.Ar(50,95.91822194),new e.Ar(51,96.916478921),new e.Ar(52,97.912720751),new e.Ar(53,98.911767757),new e.Ar(54,99.908504596),new e.Ar(55,100.908289144),new e.Ar(56,101.905607716),new e.Ar(57,102.906087204),new e.Ar(58,103.904034912),new e.Ar(59,104.905084046),new e.Ar(60,105.903483087),new e.Ar(61,106.905128453),new e.Ar(62,107.903894451),new e.Ar(63,108.905953535),new e.Ar(64,109.905152385),new e.Ar(65,110.907643952),new e.Ar(66,111.907313277),new e.Ar(67,112.910151346),new e.Ar(68,113.910365322),new e.Ar(69,114.91368341),new e.Ar(70,115.914158288),new e.Ar(71,116.91784),new e.Ar(72,117.918983915),new e.Ar(73,118.92268),new e.Ar(74,119.92403),new e.Ar(75,120.92818),new e.Ar(76,121.9298),new e.Ar(77,122.93426)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(47,93.94278),new e.Ar(48,94.93548),new e.Ar(49,95.93068),new e.Ar(50,96.924),new e.Ar(51,97.921759995),new e.Ar(52,98.917597103),new e.Ar(53,99.916069387),new e.Ar(54,100.912802135),new e.Ar(55,101.911999996),new e.Ar(56,102.908972453),new e.Ar(57,103.908628228),new e.Ar(58,104.906528234),new e.Ar(59,105.906666431),new e.Ar(60,106.90509302),new e.Ar(61,107.905953705),new e.Ar(62,108.904755514),new e.Ar(63,109.90611046),new e.Ar(64,110.905294679),new e.Ar(65,111.907004132),new e.Ar(66,112.906565708),new e.Ar(67,113.908807907),new e.Ar(68,114.908762282),new e.Ar(69,115.911359558),new e.Ar(70,116.911684187),new e.Ar(71,117.914582383),new e.Ar(72,118.915666045),new e.Ar(73,119.918788609),new e.Ar(74,120.919851074),new e.Ar(75,121.92332),new e.Ar(76,122.9249),new e.Ar(77,123.92853),new e.Ar(78,124.93054),new e.Ar(79,125.9345),new e.Ar(80,126.93688)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(48,95.93977),new e.Ar(49,96.93494),new e.Ar(50,97.927579),new e.Ar(51,98.92501),new e.Ar(52,99.920230232),new e.Ar(53,100.918681442),new e.Ar(54,101.914777255),new e.Ar(55,102.913418952),new e.Ar(56,103.909848091),new e.Ar(57,104.909467818),new e.Ar(58,105.906458007),new e.Ar(59,106.906614232),new e.Ar(60,107.904183403),new e.Ar(61,108.904985569),new e.Ar(62,109.903005578),new e.Ar(63,110.904181628),new e.Ar(64,111.902757226),new e.Ar(65,112.904400947),new e.Ar(66,113.903358121),new e.Ar(67,114.905430553),new e.Ar(68,115.904755434),new e.Ar(69,116.907218242),new e.Ar(70,117.906914144),new e.Ar(71,118.909922582),new e.Ar(72,119.909851352),new e.Ar(73,120.91298039),new e.Ar(74,121.9135),new e.Ar(75,122.917003675),new e.Ar(76,123.917648302),new e.Ar(77,124.92124717),new e.Ar(78,125.922353996),new e.Ar(79,126.926434822),new e.Ar(80,127.927760617),new e.Ar(81,128.93226),new e.Ar(82,129.93398)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(49,97.94224),new e.Ar(50,98.93461),new e.Ar(51,99.931149033),new e.Ar(52,100.92656),new e.Ar(53,101.924707541),new e.Ar(54,102.919913896),new e.Ar(55,103.918338416),new e.Ar(56,104.914673434),new e.Ar(57,105.913461134),new e.Ar(58,106.910292195),new e.Ar(59,107.909719683),new e.Ar(60,108.907154078),new e.Ar(61,109.907168783),new e.Ar(62,110.905110677),new e.Ar(63,111.905533338),new e.Ar(64,112.904061223),new e.Ar(65,113.904916758),new e.Ar(66,114.903878328),new e.Ar(67,115.905259995),new e.Ar(68,116.904515731),new e.Ar(69,117.906354623),new e.Ar(70,118.905846334),new e.Ar(71,119.907961505),new e.Ar(72,120.907848847),new e.Ar(73,121.910277103),new e.Ar(74,122.910438951),new e.Ar(75,123.913175916),new e.Ar(76,124.913601387),new e.Ar(77,125.916464532),new e.Ar(78,126.917344048),new e.Ar(79,127.920170658),new e.Ar(80,128.921657958),new e.Ar(81,129.924854941),new e.Ar(82,130.926767408),new e.Ar(83,131.932919005),new e.Ar(84,132.93834),new e.Ar(85,133.94466)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(50,99.938954),new e.Ar(51,100.93606),new e.Ar(52,101.93049),new e.Ar(53,102.92813),new e.Ar(54,103.923185469),new e.Ar(55,104.921390409),new e.Ar(56,105.916880472),new e.Ar(57,106.915666702),new e.Ar(58,107.911965339),new e.Ar(59,108.911286879),new e.Ar(60,109.907852688),new e.Ar(61,110.907735404),new e.Ar(62,111.90482081),new e.Ar(63,112.905173373),new e.Ar(64,113.902781816),new e.Ar(65,114.903345973),new e.Ar(66,115.901744149),new e.Ar(67,116.902953765),new e.Ar(68,117.901606328),new e.Ar(69,118.90330888),new e.Ar(70,119.902196571),new e.Ar(71,120.904236867),new e.Ar(72,121.903440138),new e.Ar(73,122.905721901),new e.Ar(74,123.90527463),new e.Ar(75,124.907784924),new e.Ar(76,125.907653953),new e.Ar(77,126.91035098),new e.Ar(78,127.910534953),new e.Ar(79,128.913439976),new e.Ar(80,129.913852185),new e.Ar(81,130.916919144),new e.Ar(82,131.917744455),new e.Ar(83,132.923814085),new e.Ar(84,133.928463576),new e.Ar(85,134.93473),new e.Ar(86,135.93934),new e.Ar(87,136.94579)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(52,102.94012),new e.Ar(53,103.936287),new e.Ar(54,104.931528593),new e.Ar(55,105.928183134),new e.Ar(56,106.92415),new e.Ar(57,107.92216),new e.Ar(58,108.918136092),new e.Ar(59,109.917533911),new e.Ar(60,110.912534147),new e.Ar(61,111.91239464),new e.Ar(62,112.909377941),new e.Ar(63,113.909095876),new e.Ar(64,114.906598812),new e.Ar(65,115.906797235),new e.Ar(66,116.90483959),new e.Ar(67,117.905531885),new e.Ar(68,118.90394646),new e.Ar(69,119.905074315),new e.Ar(70,120.903818044),new e.Ar(71,121.905175415),new e.Ar(72,122.904215696),new e.Ar(73,123.905937525),new e.Ar(74,124.905247804),new e.Ar(75,125.907248153),new e.Ar(76,126.906914564),new e.Ar(77,127.90916733),new e.Ar(78,128.909150092),new e.Ar(79,129.911546459),new e.Ar(80,130.911946487),new e.Ar(81,131.914413247),new e.Ar(82,132.915236466),new e.Ar(83,133.920551554),new e.Ar(84,134.925167962),new e.Ar(85,135.93066),new e.Ar(86,136.93531),new e.Ar(87,137.94096),new e.Ar(88,138.94571)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(54,105.937702),new e.Ar(55,106.935036),new e.Ar(56,107.929486838),new e.Ar(57,108.927456483),new e.Ar(58,109.922407164),new e.Ar(59,110.921120589),new e.Ar(60,111.917061617),new e.Ar(61,112.915452551),new e.Ar(62,113.912498025),new e.Ar(63,114.911578627),new e.Ar(64,115.908420253),new e.Ar(65,116.90863418),new e.Ar(66,117.905825187),new e.Ar(67,118.90640811),new e.Ar(68,119.904019891),new e.Ar(69,120.904929815),new e.Ar(70,121.903047064),new e.Ar(71,122.904272951),new e.Ar(72,123.902819466),new e.Ar(73,124.904424718),new e.Ar(74,125.903305543),new e.Ar(75,126.90521729),new e.Ar(76,127.904461383),new e.Ar(77,128.906595593),new e.Ar(78,129.906222753),new e.Ar(79,130.90852188),new e.Ar(80,131.908523782),new e.Ar(81,132.910939068),new e.Ar(82,133.911540546),new e.Ar(83,134.916450782),new e.Ar(84,135.920103155),new e.Ar(85,136.925324769),new e.Ar(86,137.92922),new e.Ar(87,138.93473),new e.Ar(88,139.9387),new e.Ar(89,140.94439),new e.Ar(90,141.9485)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(55,107.943291),new e.Ar(56,108.938191658),new e.Ar(57,109.934634181),new e.Ar(58,110.930276),new e.Ar(59,111.92797),new e.Ar(60,112.923644245),new e.Ar(61,113.92185),new e.Ar(62,114.918272),new e.Ar(63,115.916735014),new e.Ar(64,116.913647692),new e.Ar(65,117.91337523),new e.Ar(66,118.910180837),new e.Ar(67,119.910047843),new e.Ar(68,120.907366063),new e.Ar(69,121.907592451),new e.Ar(70,122.905597944),new e.Ar(71,123.906211423),new e.Ar(72,124.90462415),new e.Ar(73,125.905619387),new e.Ar(74,126.90446842),new e.Ar(75,127.905805254),new e.Ar(76,128.904987487),new e.Ar(77,129.906674018),new e.Ar(78,130.906124168),new e.Ar(79,131.907994525),new e.Ar(80,132.907806465),new e.Ar(81,133.909876552),new e.Ar(82,134.91005031),new e.Ar(83,135.914655105),new e.Ar(84,136.917872653),new e.Ar(85,137.922383666),new e.Ar(86,138.926093402),new e.Ar(87,139.93121),new e.Ar(88,140.93483),new e.Ar(89,141.94018),new e.Ar(90,142.94407),new e.Ar(91,143.94961)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(56,109.944476),new e.Ar(57,110.941632),new e.Ar(58,111.93566535),new e.Ar(59,112.933382836),new e.Ar(60,113.928145),new e.Ar(61,114.926979032),new e.Ar(62,115.921394197),new e.Ar(63,116.920564355),new e.Ar(64,117.91657092),new e.Ar(65,118.915554295),new e.Ar(66,119.91215199),new e.Ar(67,120.911386497),new e.Ar(68,121.908548396),new e.Ar(69,122.908470748),new e.Ar(70,123.905895774),new e.Ar(71,124.906398236),new e.Ar(72,125.904268868),new e.Ar(73,126.905179581),new e.Ar(74,127.903530436),new e.Ar(75,128.904779458),new e.Ar(76,129.903507903),new e.Ar(77,130.90508192),new e.Ar(78,131.904154457),new e.Ar(79,132.90590566),new e.Ar(80,133.905394504),new e.Ar(81,134.907207499),new e.Ar(82,135.907219526),new e.Ar(83,136.911562939),new e.Ar(84,137.913988549),new e.Ar(85,138.918786859),new e.Ar(86,139.921635665),new e.Ar(87,140.926646282),new e.Ar(88,141.929702981),new e.Ar(89,142.93489),new e.Ar(90,143.93823),new e.Ar(91,144.94367),new e.Ar(92,145.9473),new e.Ar(93,146.95301)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(57,111.950331),new e.Ar(58,112.944535512),new e.Ar(59,113.940841319),new e.Ar(60,114.935939),new e.Ar(61,115.932914152),new e.Ar(62,116.928639484),new e.Ar(63,117.926554883),new e.Ar(64,118.922370879),new e.Ar(65,119.920678219),new e.Ar(66,120.917183637),new e.Ar(67,121.916121946),new e.Ar(68,122.912990168),new e.Ar(69,123.912245731),new e.Ar(70,124.909724871),new e.Ar(71,125.909447953),new e.Ar(72,126.9074176),new e.Ar(73,127.907747919),new e.Ar(74,128.906063369),new e.Ar(75,129.906706163),new e.Ar(76,130.905460232),new e.Ar(77,131.906429799),new e.Ar(78,132.90544687),new e.Ar(79,133.906713419),new e.Ar(80,134.905971903),new e.Ar(81,135.907305741),new e.Ar(82,136.907083505),new e.Ar(83,137.911010537),new e.Ar(84,138.913357921),new e.Ar(85,139.917277075),new e.Ar(86,140.920043984),new e.Ar(87,141.924292317),new e.Ar(88,142.927330292),new e.Ar(89,143.932027373),new e.Ar(90,144.935388226),new e.Ar(91,145.940162028),new e.Ar(92,146.943864435),new e.Ar(93,147.948899539),new e.Ar(94,148.95272),new e.Ar(95,149.95797),new e.Ar(96,150.962)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(58,113.950941),new e.Ar(59,114.94771),new e.Ar(60,115.94168),new e.Ar(61,116.937700229),new e.Ar(62,117.93344),new e.Ar(63,118.931051927),new e.Ar(64,119.926045941),new e.Ar(65,120.924485908),new e.Ar(66,121.92026),new e.Ar(67,122.91885),new e.Ar(68,123.915088437),new e.Ar(69,124.914620234),new e.Ar(70,125.911244146),new e.Ar(71,126.911121328),new e.Ar(72,127.90830887),new e.Ar(73,128.908673749),new e.Ar(74,129.906310478),new e.Ar(75,130.906930798),new e.Ar(76,131.905056152),new e.Ar(77,132.906002368),new e.Ar(78,133.904503347),new e.Ar(79,134.905682749),new e.Ar(80,135.904570109),new e.Ar(81,136.905821414),new e.Ar(82,137.905241273),new e.Ar(83,138.908835384),new e.Ar(84,139.910599485),new e.Ar(85,140.914406439),new e.Ar(86,141.916448175),new e.Ar(87,142.920617184),new e.Ar(88,143.922940468),new e.Ar(89,144.926923807),new e.Ar(90,145.930106645),new e.Ar(91,146.933992519),new e.Ar(92,147.937682377),new e.Ar(93,148.94246),new e.Ar(94,149.94562),new e.Ar(95,150.9507),new e.Ar(96,151.95416),new e.Ar(97,152.95961)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(60,116.95001),new e.Ar(61,117.94657),new e.Ar(62,118.94099),new e.Ar(63,119.93807),new e.Ar(64,120.93301),new e.Ar(65,121.93071),new e.Ar(66,122.92624),new e.Ar(67,123.92453),new e.Ar(68,124.92067),new e.Ar(69,125.91937),new e.Ar(70,126.91616),new e.Ar(71,127.91544794),new e.Ar(72,128.912667334),new e.Ar(73,129.91232),new e.Ar(74,130.910108489),new e.Ar(75,131.910110399),new e.Ar(76,132.908396372),new e.Ar(77,133.908489607),new e.Ar(78,134.906971003),new e.Ar(79,135.907651181),new e.Ar(80,136.906465656),new e.Ar(81,137.907106826),new e.Ar(82,138.90634816),new e.Ar(83,139.909472552),new e.Ar(84,140.910957016),new e.Ar(85,141.914074489),new e.Ar(86,142.916058646),new e.Ar(87,143.919591666),new e.Ar(88,144.92163837),new e.Ar(89,145.925700146),new e.Ar(90,146.927819639),new e.Ar(91,147.932191197),new e.Ar(92,148.93437),new e.Ar(93,149.93857),new e.Ar(94,150.94156),new e.Ar(95,151.94611),new e.Ar(96,152.94945),new e.Ar(97,153.9544),new e.Ar(98,154.95813)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(61,118.95276),new e.Ar(62,119.94664),new e.Ar(63,120.94367),new e.Ar(64,121.93801),new e.Ar(65,122.93551),new e.Ar(66,123.93052),new e.Ar(67,124.92854),new e.Ar(68,125.9241),new e.Ar(69,126.92275),new e.Ar(70,127.91887),new e.Ar(71,128.918679183),new e.Ar(72,129.914339361),new e.Ar(73,130.914424137),new e.Ar(74,131.91149),new e.Ar(75,132.91155),new e.Ar(76,133.909026379),new e.Ar(77,134.909145555),new e.Ar(78,135.907143574),new e.Ar(79,136.907777634),new e.Ar(80,137.905985574),new e.Ar(81,138.906646605),new e.Ar(82,139.905434035),new e.Ar(83,140.908271103),new e.Ar(84,141.909239733),new e.Ar(85,142.912381158),new e.Ar(86,143.913642686),new e.Ar(87,144.917227871),new e.Ar(88,145.918689722),new e.Ar(89,146.922510962),new e.Ar(90,147.924394738),new e.Ar(91,148.928289207),new e.Ar(92,149.930226399),new e.Ar(93,150.93404),new e.Ar(94,151.93638),new e.Ar(95,152.94058),new e.Ar(96,153.94332),new e.Ar(97,154.94804),new e.Ar(98,155.95126),new e.Ar(99,156.95634)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(62,120.955364),new e.Ar(63,121.95165),new e.Ar(64,122.94596),new e.Ar(65,123.94296),new e.Ar(66,124.93783),new e.Ar(67,125.93531),new e.Ar(68,126.93083),new e.Ar(69,127.9288),new e.Ar(70,128.92486),new e.Ar(71,129.92338),new e.Ar(72,130.920060245),new e.Ar(73,131.91912),new e.Ar(74,132.9162),new e.Ar(75,133.915672),new e.Ar(76,134.91313914),new e.Ar(77,135.912646935),new e.Ar(78,136.910678351),new e.Ar(79,137.910748891),new e.Ar(80,138.908932181),new e.Ar(81,139.909071204),new e.Ar(82,140.907647726),new e.Ar(83,141.910039865),new e.Ar(84,142.910812233),new e.Ar(85,143.913300595),new e.Ar(86,144.914506897),new e.Ar(87,145.917588016),new e.Ar(88,146.918979001),new e.Ar(89,147.922183237),new e.Ar(90,148.923791056),new e.Ar(91,149.926995031),new e.Ar(92,150.928227869),new e.Ar(93,151.9316),new e.Ar(94,152.93365),new e.Ar(95,153.93739),new e.Ar(96,154.93999),new e.Ar(97,155.94412),new e.Ar(98,156.94717),new e.Ar(99,157.95178),new e.Ar(100,158.95523)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(66,125.94307),new e.Ar(67,126.9405),new e.Ar(68,127.93539),new e.Ar(69,128.932385),new e.Ar(70,129.92878),new e.Ar(71,130.927102697),new e.Ar(72,131.92312),new e.Ar(73,132.92221),new e.Ar(74,133.918645),new e.Ar(75,134.91824),new e.Ar(76,135.915020542),new e.Ar(77,136.91463973),new e.Ar(78,137.91291745),new e.Ar(79,138.91192415),new e.Ar(80,139.909309824),new e.Ar(81,140.9096048),new e.Ar(82,141.907718643),new e.Ar(83,142.909809626),new e.Ar(84,143.910082629),new e.Ar(85,144.912568847),new e.Ar(86,145.913112139),new e.Ar(87,146.916095794),new e.Ar(88,147.916888516),new e.Ar(89,148.92014419),new e.Ar(90,149.920886563),new e.Ar(91,150.923824739),new e.Ar(92,151.924682428),new e.Ar(93,152.927694534),new e.Ar(94,153.929483295),new e.Ar(95,154.932629551),new e.Ar(96,155.9352),new e.Ar(97,156.93927),new e.Ar(98,157.94187),new e.Ar(99,158.94639),new e.Ar(100,159.94939),new e.Ar(101,160.95433)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(67,127.94826),new e.Ar(68,128.94316),new e.Ar(69,129.94045),new e.Ar(70,130.9358),new e.Ar(71,131.93375),new e.Ar(72,132.92972),new e.Ar(73,133.92849),new e.Ar(74,134.924617),new e.Ar(75,135.923447865),new e.Ar(76,136.920713),new e.Ar(77,137.920432261),new e.Ar(78,138.916759814),new e.Ar(79,139.915801649),new e.Ar(80,140.913606636),new e.Ar(81,141.912950738),new e.Ar(82,142.910927571),new e.Ar(83,143.912585768),new e.Ar(84,144.912743879),new e.Ar(85,145.914692165),new e.Ar(86,146.915133898),new e.Ar(87,147.917467786),new e.Ar(88,148.918329195),new e.Ar(89,149.920979477),new e.Ar(90,150.921202693),new e.Ar(91,151.923490557),new e.Ar(92,152.924113189),new e.Ar(93,153.926547019),new e.Ar(94,154.928097047),new e.Ar(95,155.931060357),new e.Ar(96,156.9332),new e.Ar(97,157.93669),new e.Ar(98,158.93913),new e.Ar(99,159.94299),new e.Ar(100,160.94586),new e.Ar(101,161.95029),new e.Ar(102,162.95352)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(68,129.94863),new e.Ar(69,130.94589),new e.Ar(70,131.94082),new e.Ar(71,132.93873),new e.Ar(72,133.93402),new e.Ar(73,134.93235),new e.Ar(74,135.9283),new e.Ar(75,136.927046709),new e.Ar(76,137.92354),new e.Ar(77,138.922302),new e.Ar(78,139.918991),new e.Ar(79,140.918468512),new e.Ar(80,141.915193274),new e.Ar(81,142.914623555),new e.Ar(82,143.91199473),new e.Ar(83,144.913405611),new e.Ar(84,145.91303676),new e.Ar(85,146.914893275),new e.Ar(86,147.914817914),new e.Ar(87,148.917179521),new e.Ar(88,149.917271454),new e.Ar(89,150.919928351),new e.Ar(90,151.919728244),new e.Ar(91,152.922093907),new e.Ar(92,153.922205303),new e.Ar(93,154.92463594),new e.Ar(94,155.925526236),new e.Ar(95,156.928354506),new e.Ar(96,157.929987938),new e.Ar(97,158.9332),new e.Ar(98,159.93514),new e.Ar(99,160.93883),new e.Ar(100,161.94122),new e.Ar(101,162.94536),new e.Ar(102,163.94828),new e.Ar(103,164.95298)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(69,131.95416),new e.Ar(70,132.9489),new e.Ar(71,133.94632),new e.Ar(72,134.94172),new e.Ar(73,135.9395),new e.Ar(74,136.93521),new e.Ar(75,137.93345),new e.Ar(76,138.92882915),new e.Ar(77,139.928083921),new e.Ar(78,140.924885867),new e.Ar(79,141.923400033),new e.Ar(80,142.920286634),new e.Ar(81,143.918774116),new e.Ar(82,144.916261285),new e.Ar(83,145.917199714),new e.Ar(84,146.916741206),new e.Ar(85,147.918153775),new e.Ar(86,148.917925922),new e.Ar(87,149.919698294),new e.Ar(88,150.919846022),new e.Ar(89,151.921740399),new e.Ar(90,152.921226219),new e.Ar(91,153.922975386),new e.Ar(92,154.922889429),new e.Ar(93,155.924750855),new e.Ar(94,156.925419435),new e.Ar(95,157.927841923),new e.Ar(96,158.9290845),new e.Ar(97,159.931460406),new e.Ar(98,160.93368),new e.Ar(99,161.93704),new e.Ar(100,162.93921),new e.Ar(101,163.94299),new e.Ar(102,164.94572),new e.Ar(103,165.94997),new e.Ar(104,166.95305)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(72,135.94707),new e.Ar(73,136.94465),new e.Ar(74,137.93997),new e.Ar(75,138.93808),new e.Ar(76,139.933236934),new e.Ar(77,140.93221),new e.Ar(78,141.927908919),new e.Ar(79,142.926738636),new e.Ar(80,143.923390357),new e.Ar(81,144.921687498),new e.Ar(82,145.918305344),new e.Ar(83,146.919089446),new e.Ar(84,147.918109771),new e.Ar(85,148.919336427),new e.Ar(86,149.918655455),new e.Ar(87,150.920344273),new e.Ar(88,151.919787882),new e.Ar(89,152.921746283),new e.Ar(90,153.920862271),new e.Ar(91,154.922618801),new e.Ar(92,155.922119552),new e.Ar(93,156.923956686),new e.Ar(94,157.924100533),new e.Ar(95,158.926385075),new e.Ar(96,159.927050616),new e.Ar(97,160.929665688),new e.Ar(98,161.930981211),new e.Ar(99,162.93399),new e.Ar(100,163.93586),new e.Ar(101,164.93938),new e.Ar(102,165.9416),new e.Ar(103,166.94557),new e.Ar(104,167.94836),new e.Ar(105,168.95287)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(73,137.95287),new e.Ar(74,138.94803),new e.Ar(75,139.945367985),new e.Ar(76,140.94116),new e.Ar(77,141.939073781),new e.Ar(78,142.93475),new e.Ar(79,143.93253),new e.Ar(80,144.92888),new e.Ar(81,145.927180629),new e.Ar(82,146.924037176),new e.Ar(83,147.924298636),new e.Ar(84,148.92324163),new e.Ar(85,149.923654158),new e.Ar(86,150.923098169),new e.Ar(87,151.924071324),new e.Ar(88,152.923430858),new e.Ar(89,153.924686236),new e.Ar(90,154.923500411),new e.Ar(91,155.924743749),new e.Ar(92,156.924021155),new e.Ar(93,157.92541026),new e.Ar(94,158.925343135),new e.Ar(95,159.927164021),new e.Ar(96,160.927566289),new e.Ar(97,161.929484803),new e.Ar(98,162.930643942),new e.Ar(99,163.933347253),new e.Ar(100,164.93488),new e.Ar(101,165.93805),new e.Ar(102,166.94005),new e.Ar(103,167.94364),new e.Ar(104,168.94622),new e.Ar(105,169.95025),new e.Ar(106,170.9533)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(74,139.95379),new e.Ar(75,140.95119),new e.Ar(76,141.946695946),new e.Ar(77,142.94383),new e.Ar(78,143.93907),new e.Ar(79,144.936717),new e.Ar(80,145.932720118),new e.Ar(81,146.930878496),new e.Ar(82,147.927177882),new e.Ar(83,148.927333981),new e.Ar(84,149.925579728),new e.Ar(85,150.92617963),new e.Ar(86,151.924713874),new e.Ar(87,152.925760865),new e.Ar(88,153.924422046),new e.Ar(89,154.92574895),new e.Ar(90,155.924278273),new e.Ar(91,156.925461256),new e.Ar(92,157.924404637),new e.Ar(93,158.92573566),new e.Ar(94,159.925193718),new e.Ar(95,160.926929595),new e.Ar(96,161.926794731),new e.Ar(97,162.928727532),new e.Ar(98,163.929171165),new e.Ar(99,164.931699828),new e.Ar(100,165.932803241),new e.Ar(101,166.935649025),new e.Ar(102,167.93723),new e.Ar(103,168.940303648),new e.Ar(104,169.94267),new e.Ar(105,170.94648),new e.Ar(106,171.94911),new e.Ar(107,172.95344)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(75,141.95986),new e.Ar(76,142.95469),new e.Ar(77,143.95164),new e.Ar(78,144.94688),new e.Ar(79,145.9441),new e.Ar(80,146.93984),new e.Ar(81,147.937269),new e.Ar(82,148.933789944),new e.Ar(83,149.932760914),new e.Ar(84,150.931680791),new e.Ar(85,151.931740598),new e.Ar(86,152.930194506),new e.Ar(87,153.930596268),new e.Ar(88,154.929079084),new e.Ar(89,155.929001869),new e.Ar(90,156.928188059),new e.Ar(91,157.92894573),new e.Ar(92,158.927708537),new e.Ar(93,159.928725679),new e.Ar(94,160.927851662),new e.Ar(95,161.92909242),new e.Ar(96,162.928730286),new e.Ar(97,163.930230577),new e.Ar(98,164.930319169),new e.Ar(99,165.932281267),new e.Ar(100,166.933126195),new e.Ar(101,167.935496424),new e.Ar(102,168.936868306),new e.Ar(103,169.939614951),new e.Ar(104,170.941461227),new e.Ar(105,171.94482),new e.Ar(106,172.94729),new e.Ar(107,173.95115),new e.Ar(108,174.95405)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(76,143.96059),new e.Ar(77,144.95746),new e.Ar(78,145.95212),new e.Ar(79,146.94931),new e.Ar(80,147.94444),new e.Ar(81,148.942780527),new e.Ar(82,149.937171034),new e.Ar(83,150.93746),new e.Ar(84,151.935078452),new e.Ar(85,152.935093125),new e.Ar(86,153.932777294),new e.Ar(87,154.933204273),new e.Ar(88,155.931015001),new e.Ar(89,156.931945517),new e.Ar(90,157.929912),new e.Ar(91,158.930680718),new e.Ar(92,159.929078924),new e.Ar(93,160.930001348),new e.Ar(94,161.928774923),new e.Ar(95,162.930029273),new e.Ar(96,163.929196996),new e.Ar(97,164.9307228),new e.Ar(98,165.93028997),new e.Ar(99,166.932045448),new e.Ar(100,167.932367781),new e.Ar(101,168.934588082),new e.Ar(102,169.935460334),new e.Ar(103,170.938025885),new e.Ar(104,171.939352149),new e.Ar(105,172.9424),new e.Ar(106,173.94434),new e.Ar(107,174.94793),new e.Ar(108,175.95029),new e.Ar(109,176.95437)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(77,145.966495),new e.Ar(78,146.961081),new e.Ar(79,147.95755),new e.Ar(80,148.95265),new e.Ar(81,149.94967),new e.Ar(82,150.944842),new e.Ar(83,151.9443),new e.Ar(84,152.942027631),new e.Ar(85,153.940832325),new e.Ar(86,154.939191562),new e.Ar(87,155.939006895),new e.Ar(88,156.936756069),new e.Ar(89,157.936996),new e.Ar(90,158.934808966),new e.Ar(91,159.935090772),new e.Ar(92,160.933398042),new e.Ar(93,161.933970147),new e.Ar(94,162.932647648),new e.Ar(95,163.933450972),new e.Ar(96,164.932432463),new e.Ar(97,165.933553133),new e.Ar(98,166.932848844),new e.Ar(99,167.934170375),new e.Ar(100,168.934211117),new e.Ar(101,169.935797877),new e.Ar(102,170.936425817),new e.Ar(103,171.938396118),new e.Ar(104,172.939600336),new e.Ar(105,173.942164618),new e.Ar(106,174.943832897),new e.Ar(107,175.946991412),new e.Ar(108,176.94904),new e.Ar(109,177.95264),new e.Ar(110,178.95534)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(78,147.96676),new e.Ar(79,148.96348),new e.Ar(80,149.95799),new e.Ar(81,150.954657965),new e.Ar(82,151.950167),new e.Ar(83,152.94921),new e.Ar(84,153.945651145),new e.Ar(85,154.945792),new e.Ar(86,155.942847109),new e.Ar(87,156.94265865),new e.Ar(88,157.939857897),new e.Ar(89,158.940153735),new e.Ar(90,159.93756),new e.Ar(91,160.937357719),new e.Ar(92,161.93575),new e.Ar(93,162.936265492),new e.Ar(94,163.93452),new e.Ar(95,164.935397592),new e.Ar(96,165.933879623),new e.Ar(97,166.934946862),new e.Ar(98,167.933894465),new e.Ar(99,168.93518712),new e.Ar(100,169.934758652),new e.Ar(101,170.936322297),new e.Ar(102,171.936377696),new e.Ar(103,172.938206756),new e.Ar(104,173.938858101),new e.Ar(105,174.941272494),new e.Ar(106,175.942568409),new e.Ar(107,176.945257126),new e.Ar(108,177.946643396),new e.Ar(109,178.95017),new e.Ar(110,179.95233),new e.Ar(111,180.95615)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(79,149.972668),new e.Ar(80,150.967147),new e.Ar(81,151.96361),new e.Ar(82,152.95869),new e.Ar(83,153.9571),new e.Ar(84,154.953641324),new e.Ar(85,155.952907),new e.Ar(86,156.950101536),new e.Ar(87,157.948577981),new e.Ar(88,158.946615113),new e.Ar(89,159.945383),new e.Ar(90,160.943047504),new e.Ar(91,161.943222),new e.Ar(92,162.941203796),new e.Ar(93,163.941215),new e.Ar(94,164.939605886),new e.Ar(95,165.939762646),new e.Ar(96,166.938307056),new e.Ar(97,167.938698576),new e.Ar(98,168.937648757),new e.Ar(99,169.93847219),new e.Ar(100,170.937909903),new e.Ar(101,171.939082239),new e.Ar(102,172.938926901),new e.Ar(103,173.940333522),new e.Ar(104,174.940767904),new e.Ar(105,175.942682399),new e.Ar(106,176.943754987),new e.Ar(107,177.945951366),new e.Ar(108,178.947324216),new e.Ar(109,179.949879968),new e.Ar(110,180.95197),new e.Ar(111,181.95521),new e.Ar(112,182.95757),new e.Ar(113,183.96117)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(82,153.96425),new e.Ar(83,154.96276),new e.Ar(84,155.959247),new e.Ar(85,156.958127),new e.Ar(86,157.95405528),new e.Ar(87,158.954003),new e.Ar(88,159.950713588),new e.Ar(89,160.950330852),new e.Ar(90,161.947202977),new e.Ar(91,162.947057),new e.Ar(92,163.944422),new e.Ar(93,164.94454),new e.Ar(94,165.94225),new e.Ar(95,166.9426),new e.Ar(96,167.94063),new e.Ar(97,168.941158567),new e.Ar(98,169.93965),new e.Ar(99,170.94049),new e.Ar(100,171.93945798),new e.Ar(101,172.94065),new e.Ar(102,173.940040159),new e.Ar(103,174.941502991),new e.Ar(104,175.941401828),new e.Ar(105,176.943220013),new e.Ar(106,177.943697732),new e.Ar(107,178.945815073),new e.Ar(108,179.94654876),new e.Ar(109,180.949099124),new e.Ar(110,181.950552893),new e.Ar(111,182.953531012),new e.Ar(112,183.95544788),new e.Ar(113,184.95878),new e.Ar(114,185.96092)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(83,155.971689),new e.Ar(84,156.968145),new e.Ar(85,157.966368),new e.Ar(86,158.96232309),new e.Ar(87,159.961358),new e.Ar(88,160.958372992),new e.Ar(89,161.956556553),new e.Ar(90,162.95431665),new e.Ar(91,163.95357),new e.Ar(92,164.950817),new e.Ar(93,165.95047),new e.Ar(94,166.948639),new e.Ar(95,167.947787),new e.Ar(96,168.94592),new e.Ar(97,169.94609),new e.Ar(98,170.94446),new e.Ar(99,171.944739818),new e.Ar(100,172.94459),new e.Ar(101,173.944167937),new e.Ar(102,174.94365),new e.Ar(103,175.944740551),new e.Ar(104,176.944471766),new e.Ar(105,177.945750349),new e.Ar(106,178.945934113),new e.Ar(107,179.947465655),new e.Ar(108,180.947996346),new e.Ar(109,181.950152414),new e.Ar(110,182.951373188),new e.Ar(111,183.954009331),new e.Ar(112,184.955559086),new e.Ar(113,185.9585501),new e.Ar(114,186.96041),new e.Ar(115,187.96371)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(84,157.973939),new e.Ar(85,158.97228),new e.Ar(86,159.968369),new e.Ar(87,160.967089),new e.Ar(88,161.962750303),new e.Ar(89,162.962532),new e.Ar(90,163.95898381),new e.Ar(91,164.958335962),new e.Ar(92,165.955019896),new e.Ar(93,166.954672),new e.Ar(94,167.951863),new e.Ar(95,168.951759),new e.Ar(96,169.948473988),new e.Ar(97,170.94946),new e.Ar(98,171.948228837),new e.Ar(99,172.948884),new e.Ar(100,173.94616),new e.Ar(101,174.94677),new e.Ar(102,175.94559),new e.Ar(103,176.94662),new e.Ar(104,177.945848364),new e.Ar(105,178.947071733),new e.Ar(106,179.946705734),new e.Ar(107,180.948198054),new e.Ar(108,181.948205519),new e.Ar(109,182.950224458),new e.Ar(110,183.950932553),new e.Ar(111,184.953420586),new e.Ar(112,185.954362204),new e.Ar(113,186.957158365),new e.Ar(114,187.958486954),new e.Ar(115,188.96191222),new e.Ar(116,189.963179541)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(85,159.981485),new e.Ar(86,160.977661),new e.Ar(87,161.975707),new e.Ar(88,162.971375872),new e.Ar(89,163.970319),new e.Ar(90,164.967050268),new e.Ar(91,165.965211372),new e.Ar(92,166.962564),new e.Ar(93,167.961609),new e.Ar(94,168.95883),new e.Ar(95,169.958163),new e.Ar(96,170.955547),new e.Ar(97,171.955285),new e.Ar(98,172.953062),new e.Ar(99,173.952114),new e.Ar(100,174.951393),new e.Ar(101,175.95157),new e.Ar(102,176.95027),new e.Ar(103,177.950851081),new e.Ar(104,178.949981038),new e.Ar(105,179.95078768),new e.Ar(106,180.950064596),new e.Ar(107,181.951211444),new e.Ar(108,182.950821349),new e.Ar(109,183.952524289),new e.Ar(110,184.952955747),new e.Ar(111,185.954986529),new e.Ar(112,186.955750787),new e.Ar(113,187.958112287),new e.Ar(114,188.959228359),new e.Ar(115,189.961816139),new e.Ar(116,190.963123592),new e.Ar(117,191.96596)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(86,161.983819),new e.Ar(87,162.982048),new e.Ar(88,163.977927),new e.Ar(89,164.976475),new e.Ar(90,165.971934911),new e.Ar(91,166.971554),new e.Ar(92,167.967832911),new e.Ar(93,168.967076205),new e.Ar(94,169.963569716),new e.Ar(95,170.96304),new e.Ar(96,171.960078),new e.Ar(97,172.959791),new e.Ar(98,173.956307704),new e.Ar(99,174.95708),new e.Ar(100,175.953757941),new e.Ar(101,176.955045),new e.Ar(102,177.953348225),new e.Ar(103,178.953951),new e.Ar(104,179.952308241),new e.Ar(105,180.953274494),new e.Ar(106,181.952186222),new e.Ar(107,182.95311),new e.Ar(108,183.952490808),new e.Ar(109,184.954043023),new e.Ar(110,185.953838355),new e.Ar(111,186.955747928),new e.Ar(112,187.955835993),new e.Ar(113,188.958144866),new e.Ar(114,189.95844521),new e.Ar(115,190.960927951),new e.Ar(116,191.961479047),new e.Ar(117,192.964148083),new e.Ar(118,193.965179314),new e.Ar(119,194.968123889),new e.Ar(120,195.96962255)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(88,164.98758),new e.Ar(89,165.985506),new e.Ar(90,166.980951577),new e.Ar(91,167.979966),new e.Ar(92,168.976390868),new e.Ar(93,169.974441697),new e.Ar(94,170.971779),new e.Ar(95,171.970643),new e.Ar(96,172.967707),new e.Ar(97,173.966804),new e.Ar(98,174.964279),new e.Ar(99,175.963511),new e.Ar(100,176.96117),new e.Ar(101,177.960084944),new e.Ar(102,178.95915),new e.Ar(103,179.958555615),new e.Ar(104,180.957642156),new e.Ar(105,181.958127689),new e.Ar(106,182.956814),new e.Ar(107,183.957388318),new e.Ar(108,184.95659),new e.Ar(109,185.957951104),new e.Ar(110,186.95736083),new e.Ar(111,187.958851962),new e.Ar(112,188.958716473),new e.Ar(113,189.960592299),new e.Ar(114,190.960591191),new e.Ar(115,191.962602198),new e.Ar(116,192.9629237),new e.Ar(117,193.96507561),new e.Ar(118,194.9659768),new e.Ar(119,195.968379906),new e.Ar(120,196.969636496),new e.Ar(121,197.97228),new e.Ar(122,198.973787159)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(90,167.988035),new e.Ar(91,168.986421),new e.Ar(92,169.981734918),new e.Ar(93,170.981251),new e.Ar(94,171.977376138),new e.Ar(95,172.976499642),new e.Ar(96,173.972811276),new e.Ar(97,174.972276),new e.Ar(98,175.969),new e.Ar(99,176.968453),new e.Ar(100,177.964894223),new e.Ar(101,178.965475),new e.Ar(102,179.962023729),new e.Ar(103,180.963177),new e.Ar(104,181.961267637),new e.Ar(105,182.961729),new e.Ar(106,183.959851685),new e.Ar(107,184.960753782),new e.Ar(108,185.959432346),new e.Ar(109,186.960697),new e.Ar(110,187.959395697),new e.Ar(111,188.9608319),new e.Ar(112,189.959930073),new e.Ar(113,190.961684653),new e.Ar(114,191.961035158),new e.Ar(115,192.962984504),new e.Ar(116,193.962663581),new e.Ar(117,194.964774449),new e.Ar(118,195.964934884),new e.Ar(119,196.967323401),new e.Ar(120,197.967876009),new e.Ar(121,198.970576213),new e.Ar(122,199.971423885),new e.Ar(123,200.974496467),new e.Ar(124,201.97574)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(92,170.991183),new e.Ar(93,171.990109),new e.Ar(94,172.986398138),new e.Ar(95,173.984325861),new e.Ar(96,174.981552),new e.Ar(97,175.980269),new e.Ar(98,176.977215),new e.Ar(99,177.975975),new e.Ar(100,178.973412),new e.Ar(101,179.972396),new e.Ar(102,180.969948),new e.Ar(103,181.968621416),new e.Ar(104,182.96762),new e.Ar(105,183.966776046),new e.Ar(106,184.965806956),new e.Ar(107,185.965997671),new e.Ar(108,186.964562),new e.Ar(109,187.965321662),new e.Ar(110,188.9642243),new e.Ar(111,189.964698757),new e.Ar(112,190.963649239),new e.Ar(113,191.964810107),new e.Ar(114,192.964131745),new e.Ar(115,193.96533889),new e.Ar(116,194.965017928),new e.Ar(117,195.966551315),new e.Ar(118,196.966551609),new e.Ar(119,197.968225244),new e.Ar(120,198.968748016),new e.Ar(121,199.970717886),new e.Ar(122,200.971640839),new e.Ar(123,201.973788431),new e.Ar(124,202.975137256),new e.Ar(125,203.977705),new e.Ar(126,204.97961)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(95,174.991411),new e.Ar(96,175.987413248),new e.Ar(97,176.986336874),new e.Ar(98,177.982476325),new e.Ar(99,178.981783),new e.Ar(100,179.978322),new e.Ar(101,180.977806),new e.Ar(102,181.97393546),new e.Ar(103,182.974561),new e.Ar(104,183.970705219),new e.Ar(105,184.971983),new e.Ar(106,185.969460021),new e.Ar(107,186.969785),new e.Ar(108,187.967511693),new e.Ar(109,188.968733187),new e.Ar(110,189.966958568),new e.Ar(111,190.96706311),new e.Ar(112,191.965921572),new e.Ar(113,192.966644169),new e.Ar(114,193.965381832),new e.Ar(115,194.966638981),new e.Ar(116,195.965814846),new e.Ar(117,196.967195333),new e.Ar(118,197.96675183),new e.Ar(119,198.968262489),new e.Ar(120,199.968308726),new e.Ar(121,200.970285275),new e.Ar(122,201.970625604),new e.Ar(123,202.972857096),new e.Ar(124,203.97347564),new e.Ar(125,204.976056104),new e.Ar(126,205.977498672),new e.Ar(127,206.982577025),new e.Ar(128,207.98594)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(96,176.996881),new e.Ar(97,177.994637),new e.Ar(98,178.991466),new e.Ar(99,179.990194),new e.Ar(100,180.986904),new e.Ar(101,181.98561),new e.Ar(102,182.982697),new e.Ar(103,183.98176),new e.Ar(104,184.9791),new e.Ar(105,185.977549881),new e.Ar(106,186.97617),new e.Ar(107,187.97592),new e.Ar(108,188.974290451),new e.Ar(109,189.974473379),new e.Ar(110,190.972261952),new e.Ar(111,191.972770785),new e.Ar(112,192.970548),new e.Ar(113,193.971053),new e.Ar(114,194.96965),new e.Ar(115,195.970515),new e.Ar(116,196.9695362),new e.Ar(117,197.970466294),new e.Ar(118,198.969813837),new e.Ar(119,199.970945394),new e.Ar(120,200.97080377),new e.Ar(121,201.972090569),new e.Ar(122,202.972329088),new e.Ar(123,203.973848646),new e.Ar(124,204.97441227),new e.Ar(125,205.976095321),new e.Ar(126,206.977407908),new e.Ar(127,207.982004653),new e.Ar(128,208.985349125),new e.Ar(129,209.990065574)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(99,180.996714),new e.Ar(100,181.992676101),new e.Ar(101,182.99193),new e.Ar(102,183.988198),new e.Ar(103,184.98758),new e.Ar(104,185.983485388),new e.Ar(105,186.98403),new e.Ar(106,187.979869108),new e.Ar(107,188.98088),new e.Ar(108,189.978180008),new e.Ar(109,190.9782),new e.Ar(110,191.975719811),new e.Ar(111,192.97608),new e.Ar(112,193.974648056),new e.Ar(113,194.975920279),new e.Ar(114,195.97271),new e.Ar(115,196.97338),new e.Ar(116,197.97198),new e.Ar(117,198.972909384),new e.Ar(118,199.97181556),new e.Ar(119,200.972846589),new e.Ar(120,201.972143786),new e.Ar(121,202.973375491),new e.Ar(122,203.973028761),new e.Ar(123,204.974467112),new e.Ar(124,205.974449002),new e.Ar(125,206.975880605),new e.Ar(126,207.97663585),new e.Ar(127,208.981074801),new e.Ar(128,209.984173129),new e.Ar(129,210.988731474),new e.Ar(130,211.991887495),new e.Ar(131,212.9965),new e.Ar(132,213.999798147)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(102,184.997708),new e.Ar(103,185.99648),new e.Ar(104,186.993458),new e.Ar(105,187.992173),new e.Ar(106,188.989505),new e.Ar(107,189.987520007),new e.Ar(108,190.986053),new e.Ar(109,191.985368),new e.Ar(110,192.983662229),new e.Ar(111,193.983430186),new e.Ar(112,194.98112697),new e.Ar(113,195.981236107),new e.Ar(114,196.978934287),new e.Ar(115,197.979024396),new e.Ar(116,198.977576953),new e.Ar(117,199.978141983),new e.Ar(118,200.976970721),new e.Ar(119,201.977674504),new e.Ar(120,202.976868118),new e.Ar(121,203.977805161),new e.Ar(122,204.977374688),new e.Ar(123,205.978482854),new e.Ar(124,206.978455217),new e.Ar(125,207.979726699),new e.Ar(126,208.980383241),new e.Ar(127,209.984104944),new e.Ar(128,210.987258139),new e.Ar(129,211.991271542),new e.Ar(130,212.994374836),new e.Ar(131,213.998698664),new e.Ar(132,215.001832349),new e.Ar(133,216.006199)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(106,189.994293888),new e.Ar(107,190.994653),new e.Ar(108,191.99033039),new e.Ar(109,192.991102),new e.Ar(110,193.988284107),new e.Ar(111,194.988045),new e.Ar(112,195.985469432),new e.Ar(113,196.985567),new e.Ar(114,197.984024384),new e.Ar(115,198.985044507),new e.Ar(116,199.981735),new e.Ar(117,200.982209),new e.Ar(118,201.980704),new e.Ar(119,202.981412863),new e.Ar(120,203.980307113),new e.Ar(121,204.981165396),new e.Ar(122,205.980465241),new e.Ar(123,206.981578228),new e.Ar(124,207.981231059),new e.Ar(125,208.982415788),new e.Ar(126,209.982857396),new e.Ar(127,210.986636869),new e.Ar(128,211.988851755),new e.Ar(129,212.992842522),new e.Ar(130,213.995185949),new e.Ar(131,214.999414609),new e.Ar(132,216.001905198),new e.Ar(133,217.006253),new e.Ar(134,218.008965773)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(108,193.000188),new e.Ar(109,193.997973),new e.Ar(110,194.996554),new e.Ar(111,195.995702),new e.Ar(112,196.993891293),new e.Ar(113,197.99343368),new e.Ar(114,198.991008569),new e.Ar(115,199.990920883),new e.Ar(116,200.988486908),new e.Ar(117,201.988448629),new e.Ar(118,202.986847216),new e.Ar(119,203.987261559),new e.Ar(120,204.986036352),new e.Ar(121,205.986599242),new e.Ar(122,206.985775861),new e.Ar(123,207.986582508),new e.Ar(124,208.986158678),new e.Ar(125,209.987131308),new e.Ar(126,210.987480806),new e.Ar(127,211.990734657),new e.Ar(128,212.99292115),new e.Ar(129,213.996356412),new e.Ar(130,214.998641245),new e.Ar(131,216.002408839),new e.Ar(132,217.004709619),new e.Ar(133,218.008681458),new e.Ar(134,219.011296478),new e.Ar(135,220.015301),new e.Ar(136,221.01814),new e.Ar(137,222.02233),new e.Ar(138,223.02534)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(110,196.001117268),new e.Ar(111,197.001661),new e.Ar(112,197.998779978),new e.Ar(113,198.998309),new e.Ar(114,199.995634148),new e.Ar(115,200.995535),new e.Ar(116,201.993899382),new e.Ar(117,202.994765192),new e.Ar(118,203.991365),new e.Ar(119,204.991668),new e.Ar(120,205.99016),new e.Ar(121,206.990726826),new e.Ar(122,207.989631237),new e.Ar(123,208.990376634),new e.Ar(124,209.989679862),new e.Ar(125,210.99058541),new e.Ar(126,211.990688899),new e.Ar(127,212.993868354),new e.Ar(128,213.995346275),new e.Ar(129,214.998729195),new e.Ar(130,216.000258153),new e.Ar(131,217.003914555),new e.Ar(132,218.005586315),new e.Ar(133,219.009474831),new e.Ar(134,220.011384149),new e.Ar(135,221.015455),new e.Ar(136,222.017570472),new e.Ar(137,223.02179),new e.Ar(138,224.02409),new e.Ar(139,225.02844),new e.Ar(140,226.03089),new e.Ar(141,227.035407),new e.Ar(142,228.038084)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(113,200.006499),new e.Ar(114,201.00458692),new e.Ar(115,202.00396885),new e.Ar(116,203.001423829),new e.Ar(117,204.001221209),new e.Ar(118,204.998663961),new e.Ar(119,205.998486886),new e.Ar(120,206.996859385),new e.Ar(121,207.997133849),new e.Ar(122,208.995915421),new e.Ar(123,209.996398327),new e.Ar(124,210.995529332),new e.Ar(125,211.996194988),new e.Ar(126,212.996174845),new e.Ar(127,213.99895474),new e.Ar(128,215.000326029),new e.Ar(129,216.003187873),new e.Ar(130,217.004616452),new e.Ar(131,218.007563326),new e.Ar(132,219.009240843),new e.Ar(133,220.012312978),new e.Ar(134,221.014245654),new e.Ar(135,222.017543957),new e.Ar(136,223.019730712),new e.Ar(137,224.023235513),new e.Ar(138,225.025606914),new e.Ar(139,226.029343423),new e.Ar(140,227.031833167),new e.Ar(141,228.034776087),new e.Ar(142,229.038426),new e.Ar(143,230.04251),new e.Ar(144,231.045407),new e.Ar(145,232.049654)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(115,203.00921),new e.Ar(116,204.006434513),new e.Ar(117,205.006187),new e.Ar(118,206.004463814),new e.Ar(119,207.005176607),new e.Ar(120,208.001776),new e.Ar(121,209.001944),new e.Ar(122,210.000446),new e.Ar(123,211.000893996),new e.Ar(124,211.999783492),new e.Ar(125,213.000345847),new e.Ar(126,214.000091141),new e.Ar(127,215.002704195),new e.Ar(128,216.003518402),new e.Ar(129,217.00630601),new e.Ar(130,218.007123948),new e.Ar(131,219.010068787),new e.Ar(132,220.011014669),new e.Ar(133,221.013907762),new e.Ar(134,222.01536182),new e.Ar(135,223.01849714),new e.Ar(136,224.020202004),new e.Ar(137,225.023604463),new e.Ar(138,226.025402555),new e.Ar(139,227.029170677),new e.Ar(140,228.031064101),new e.Ar(141,229.034820309),new e.Ar(142,230.037084774),new e.Ar(143,231.04122),new e.Ar(144,232.043693),new e.Ar(145,233.047995),new e.Ar(146,234.050547)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(118,207.012469754),new e.Ar(119,208.012112949),new e.Ar(120,209.009568736),new e.Ar(121,210.009256802),new e.Ar(122,211.007648196),new e.Ar(123,212.007811441),new e.Ar(124,213.006573689),new e.Ar(125,214.006893072),new e.Ar(126,215.006450832),new e.Ar(127,216.008721268),new e.Ar(128,217.009332676),new e.Ar(129,218.011625045),new e.Ar(130,219.012404918),new e.Ar(131,220.014752105),new e.Ar(132,221.015575746),new e.Ar(133,222.017828852),new e.Ar(134,223.01912603),new e.Ar(135,224.021708435),new e.Ar(136,225.023220576),new e.Ar(137,226.026089848),new e.Ar(138,227.027746979),new e.Ar(139,228.031014825),new e.Ar(140,229.032930871),new e.Ar(141,230.036025144),new e.Ar(142,231.038551503),new e.Ar(143,232.042022474),new e.Ar(144,233.04455),new e.Ar(145,234.04842),new e.Ar(146,235.051102),new e.Ar(147,236.055178)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(120,210.015711883),new e.Ar(121,211.016306912),new e.Ar(122,212.012916),new e.Ar(123,213.012962),new e.Ar(124,214.011451),new e.Ar(125,215.011726597),new e.Ar(126,216.011050963),new e.Ar(127,217.013066169),new e.Ar(128,218.013267744),new e.Ar(129,219.015521253),new e.Ar(130,220.015733126),new e.Ar(131,221.018171499),new e.Ar(132,222.018454131),new e.Ar(133,223.020795153),new e.Ar(134,224.02145925),new e.Ar(135,225.023941441),new e.Ar(136,226.024890681),new e.Ar(137,227.027698859),new e.Ar(138,228.028731348),new e.Ar(139,229.03175534),new e.Ar(140,230.033126574),new e.Ar(141,231.03629706),new e.Ar(142,232.03805036),new e.Ar(143,233.041576923),new e.Ar(144,234.043595497),new e.Ar(145,235.04750442),new e.Ar(146,236.04971),new e.Ar(147,237.053894),new e.Ar(148,238.056243)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(122,213.021183209),new e.Ar(123,214.02073923),new e.Ar(124,215.019097612),new e.Ar(125,216.019109649),new e.Ar(126,217.018288571),new e.Ar(127,218.020007906),new e.Ar(128,219.019880348),new e.Ar(129,220.021876493),new e.Ar(130,221.021863742),new e.Ar(131,222.023726),new e.Ar(132,223.023963748),new e.Ar(133,224.025614854),new e.Ar(134,225.026115172),new e.Ar(135,226.02793275),new e.Ar(136,227.028793151),new e.Ar(137,228.031036942),new e.Ar(138,229.032088601),new e.Ar(139,230.034532562),new e.Ar(140,231.035878898),new e.Ar(141,232.03858172),new e.Ar(142,233.040240235),new e.Ar(143,234.043302325),new e.Ar(144,235.045436759),new e.Ar(145,236.048675176),new e.Ar(146,237.05113943),new e.Ar(147,238.054497046),new e.Ar(148,239.05713),new e.Ar(149,240.06098)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(126,218.023487),new e.Ar(127,219.024915423),new e.Ar(128,220.024712),new e.Ar(129,221.026351),new e.Ar(130,222.02607),new e.Ar(131,223.027722956),new e.Ar(132,224.027590139),new e.Ar(133,225.029384369),new e.Ar(134,226.02933975),new e.Ar(135,227.031140069),new e.Ar(136,228.031366357),new e.Ar(137,229.033496137),new e.Ar(138,230.033927392),new e.Ar(139,231.036289158),new e.Ar(140,232.03714628),new e.Ar(141,233.039628196),new e.Ar(142,234.040945606),new e.Ar(143,235.043923062),new e.Ar(144,236.045561897),new e.Ar(145,237.048723955),new e.Ar(146,238.050782583),new e.Ar(147,239.054287777),new e.Ar(148,240.056585734),new e.Ar(149,241.06033),new e.Ar(150,242.062925)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(132,225.033899689),new e.Ar(133,226.035129),new e.Ar(134,227.034958261),new e.Ar(135,228.03618),new e.Ar(136,229.036246866),new e.Ar(137,230.037812591),new e.Ar(138,231.038233161),new e.Ar(139,232.040099),new e.Ar(140,233.04073235),new e.Ar(141,234.042888556),new e.Ar(142,235.044055876),new e.Ar(143,236.046559724),new e.Ar(144,237.048167253),new e.Ar(145,238.050940464),new e.Ar(146,239.052931399),new e.Ar(147,240.056168828),new e.Ar(148,241.058246266),new e.Ar(149,242.061635),new e.Ar(150,243.064273),new e.Ar(151,244.06785)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(134,228.038727686),new e.Ar(135,229.040138934),new e.Ar(136,230.039645603),new e.Ar(137,231.041258),new e.Ar(138,232.041179445),new e.Ar(139,233.04298757),new e.Ar(140,234.043304681),new e.Ar(141,235.0452815),new e.Ar(142,236.046048088),new e.Ar(143,237.048403774),new e.Ar(144,238.0495534),new e.Ar(145,239.052156519),new e.Ar(146,240.05380746),new e.Ar(147,241.056845291),new e.Ar(148,242.058736847),new e.Ar(149,243.061997013),new e.Ar(150,244.06419765),new e.Ar(151,245.067738657),new e.Ar(152,246.070198429),new e.Ar(153,247.07407)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(136,231.04556),new e.Ar(137,232.04659),new e.Ar(138,233.046472),new e.Ar(139,234.047794),new e.Ar(140,235.048029),new e.Ar(141,236.049569),new e.Ar(142,237.049970748),new e.Ar(143,238.051977839),new e.Ar(144,239.053018481),new e.Ar(145,240.055287826),new e.Ar(146,241.056822944),new e.Ar(147,242.059543039),new e.Ar(148,243.061372686),new e.Ar(149,244.064279429),new e.Ar(150,245.066445398),new e.Ar(151,246.069768438),new e.Ar(152,247.072086),new e.Ar(153,248.075745),new e.Ar(154,249.07848)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(137,233.0508),new e.Ar(138,234.05024),new e.Ar(139,235.051591),new e.Ar(140,236.051405),new e.Ar(141,237.052891),new e.Ar(142,238.053016298),new e.Ar(143,239.054951),new e.Ar(144,240.055519046),new e.Ar(145,241.057646736),new e.Ar(146,242.058829326),new e.Ar(147,243.061382249),new e.Ar(148,244.062746349),new e.Ar(149,245.065485586),new e.Ar(150,246.067217551),new e.Ar(151,247.070346811),new e.Ar(152,248.072342247),new e.Ar(153,249.075947062),new e.Ar(154,250.078350687),new e.Ar(155,251.082277873),new e.Ar(156,252.08487)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(138,235.05658),new e.Ar(139,236.05733),new e.Ar(140,237.057127),new e.Ar(141,238.058266),new e.Ar(142,239.058362),new e.Ar(143,240.059749),new e.Ar(144,241.060223),new e.Ar(145,242.06205),new e.Ar(146,243.06300157),new e.Ar(147,244.065167882),new e.Ar(148,245.066355386),new e.Ar(149,246.068666836),new e.Ar(150,247.070298533),new e.Ar(151,248.07308),new e.Ar(152,249.074979937),new e.Ar(153,250.078310529),new e.Ar(154,251.08075344),new e.Ar(155,252.084303),new e.Ar(156,253.08688),new e.Ar(157,254.0906)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(139,237.06207),new e.Ar(140,238.06141),new e.Ar(141,239.062579),new e.Ar(142,240.062295),new e.Ar(143,241.063716),new e.Ar(144,242.063688713),new e.Ar(145,243.065421),new e.Ar(146,244.06599039),new e.Ar(147,245.068039),new e.Ar(148,246.068798807),new e.Ar(149,247.070992043),new e.Ar(150,248.07217808),new e.Ar(151,249.074846818),new e.Ar(152,250.076399951),new e.Ar(153,251.079580056),new e.Ar(154,252.081619582),new e.Ar(155,253.085126791),new e.Ar(156,254.087316198),new e.Ar(157,255.091039),new e.Ar(158,256.09344)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(141,240.06892),new e.Ar(142,241.068662),new e.Ar(143,242.069699),new e.Ar(144,243.069631),new e.Ar(145,244.070969),new e.Ar(146,245.071317),new e.Ar(147,246.072965),new e.Ar(148,247.07365),new e.Ar(149,248.075458),new e.Ar(150,249.076405),new e.Ar(151,250.078654),new e.Ar(152,251.079983592),new e.Ar(153,252.082972247),new e.Ar(154,253.084817974),new e.Ar(155,254.088016026),new e.Ar(156,255.090266386),new e.Ar(157,256.093592),new e.Ar(158,257.095979)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(142,242.07343),new e.Ar(143,243.07451),new e.Ar(144,244.074077),new e.Ar(145,245.075375),new e.Ar(146,246.075281634),new e.Ar(147,247.076819),new e.Ar(148,248.077184411),new e.Ar(149,249.079024),new e.Ar(150,250.079514759),new e.Ar(151,251.081566467),new e.Ar(152,252.082460071),new e.Ar(153,253.085176259),new e.Ar(154,254.086847795),new e.Ar(155,255.089955466),new e.Ar(156,256.091766522),new e.Ar(157,257.095098635),new e.Ar(158,258.097069),new e.Ar(159,259.100588)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(144,245.081017),new e.Ar(145,246.081933),new e.Ar(146,247.081804),new e.Ar(147,248.082909),new e.Ar(148,249.083002),new e.Ar(149,250.084488),new e.Ar(150,251.084919),new e.Ar(151,252.08663),new e.Ar(152,253.08728),new e.Ar(153,254.089725),new e.Ar(154,255.091075196),new e.Ar(155,256.094052757),new e.Ar(156,257.095534643),new e.Ar(157,258.098425321),new e.Ar(158,259.100503),new e.Ar(159,260.103645)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(147,249.087823),new e.Ar(148,250.087493),new e.Ar(149,251.08896),new e.Ar(150,252.088965909),new e.Ar(151,253.090649),new e.Ar(152,254.090948746),new e.Ar(153,255.093232449),new e.Ar(154,256.094275879),new e.Ar(155,257.096852778),new e.Ar(156,258.0982),new e.Ar(157,259.101024),new e.Ar(158,260.102636),new e.Ar(159,261.105743),new e.Ar(160,262.10752)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(148,251.09436),new e.Ar(149,252.09533),new e.Ar(150,253.095258),new e.Ar(151,254.096587),new e.Ar(152,255.096769),new e.Ar(153,256.098763),new e.Ar(154,257.099606),new e.Ar(155,258.101883),new e.Ar(156,259.10299),new e.Ar(157,260.105572),new e.Ar(158,261.106941),new e.Ar(159,262.109692),new e.Ar(160,263.111394)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(149,253.100679),new e.Ar(150,254.100166),new e.Ar(151,255.101492),new e.Ar(152,256.101179573),new e.Ar(153,257.103072),new e.Ar(154,258.103568),new e.Ar(155,259.105628),new e.Ar(156,260.106434),new e.Ar(157,261.108752),new e.Ar(158,262.109918),new e.Ar(159,263.11254),new e.Ar(160,264.113978)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(150,255.107398),new e.Ar(151,256.10811),new e.Ar(152,257.107858),new e.Ar(153,258.109438),new e.Ar(154,259.109721),new e.Ar(155,260.111427),new e.Ar(156,261.112106),new e.Ar(157,262.114153),new e.Ar(158,263.115078),new e.Ar(159,264.117473),new e.Ar(160,265.118659)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(152,258.113151),new e.Ar(153,259.114652),new e.Ar(154,260.114435447),new e.Ar(155,261.116199),new e.Ar(156,262.116477),new e.Ar(157,263.118313),new e.Ar(158,264.118924),new e.Ar(159,265.121066),new e.Ar(160,266.121928)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(153,260.121803),new e.Ar(154,261.1218),new e.Ar(155,262.123009),new e.Ar(156,263.123146),new e.Ar(157,264.12473),new e.Ar(158,265.125198),new e.Ar(159,266.127009),new e.Ar(160,267.12774)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(155,263.12871),new e.Ar(156,264.128408258),new e.Ar(157,265.130001),new e.Ar(158,266.130042),new e.Ar(159,267.131774),new e.Ar(160,268.132156),new e.Ar(161,269.134114)]),v(F(l.O7,1),Vt,3,0,[new e.Ar(156,265.136567),new e.Ar(157,266.13794),new e.Ar(158,267.137526),new e.Ar(159,268.138816),new e.Ar(160,269.139106),new e.Ar(161,270.140723),new e.Ar(162,271.141229)])]);},e.Dr=function(t,A){e.Cr();var n,P;for(P=A-t,n=0;n<e.Br[t].length;n++)if(e.Br[t][n].b==P)return e.Br[t][n].a;return NaN;},e.Er=function(t,A){var n;if(t.b.length!=A.b.length)return t.b.length<A.b.length?-1:1;for(n=0;n<t.b.length;n++)if(t.b[n]!=A.b[n])return t.b[n]<A.b[n]?-1:1;return 0;},e.Fr=function(t){var A,n,P,r,i,D;for(n=0,r=t.a,i=0,D=r.length;i<D;++i)P=r[i],P&&++n;for(t.b=j(l.j7,H,6,n,15,1),n=0,A=0;A<t.a.length;A++)t.a[A]&&(t.b[n++]=A);},e.Gr=function(t,A){var n;for(e.Fr(t),n=0;n<t.b.length;n++)if(A[t.b[n]])return  true;return  false;},e.Hr=function(t){this.a=j(l.Feb,ht,6,t,16,1);},N(239,1,{239:1,32:1},e.Hr),U.Sb=function(t){return e.Er(this,t);},l.P7=O(239),e.Mr=function(){e.Mr=D1,e.Lr=v(F(l.h7,1),Ft,6,15,[0,1.00794,4.0026,6.941,9.0122,10.811,12.011,14.007,15.999,18.998,20.18,22.99,24.305,26.982,28.086,30.974,32.066,35.453,39.948,39.098,40.078,44.956,47.867,50.942,51.996,54.938,55.845,58.933,58.693,63.546,65.39,69.723,72.61,74.922,78.96,79.904,83.8,85.468,87.62,88.906,91.224,92.906,95.94,98.906,101.07,102.91,106.42,107.87,112.41,114.82,118.71,121.76,127.6,126.9,131.29,132.91,137.33,138.91,140.12,140.91,144.24,146.92,150.36,151.96,157.25,158.93,162.5,164.93,167.26,168.93,173.04,174.97,178.49,180.95,183.84,186.21,190.23,192.22,195.08,196.97,200.59,204.38,207.2,208.98,209.98,209.99,222.02,223.02,226.03,227.03,232.04,231.04,238.03,237.05,239.05,241.06,244.06,249.08,252.08,252.08,257.1,258.1,259.1,262.11,267.12,268.13,271.13,270.13,277.15,276.15,eP,eP,283.17,285.18,nP,nP,293.2,PP,PP,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.0141,3.016,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71.0787,156.18828,114.10364,115.0877,103.1447,128.13052,129.11458,57.05182,137.14158,113.15934,113.15934,128.17428,131.19846,147.17646,97.11658,87.0777,101.10458,186.2134,163.17546,99.13246]),e.Jr=v(F(l.h7,1),Ft,6,15,[0,1.007825,4.0026,7.016003,9.012182,11.009305,12,14.003074,15.994915,18.998403,19.992435,22.989767,23.985042,26.98153,27.976927,30.973762,31.97207,34.968852,39.962384,38.963707,39.962591,44.95591,47.947947,50.943962,51.940509,54.938047,55.934939,58.933198,57.935346,62.939598,63.929145,68.92558,73.921177,74.921594,79.91652,78.918336,83.911507,84.911794,87.905619,88.905849,89.904703,92.906377,97.905406,89.92381,101.904348,102.9055,105.903478,106.905092,113.903357,114.90388,119.9022,120.903821,129.906229,126.904473,131.904144,132.905429,137.905232,138.906346,139.905433,140.907647,141.907719,135.92398,151.919729,152.921225,157.924099,158.925342,163.929171,164.930319,165.93029,168.934212,173.938859,174.94077,179.946545,180.947992,183.950928,186.955744,191.961467,192.962917,194.964766,196.966543,201.970617,204.974401,207.976627,208.980374,193.98818,195.99573,199.9957,201.00411,206.0038,210.00923,232.038054,216.01896,238.050784,229.03623,232.041169,237.05005,238.05302,242.06194,240.06228,243.06947,243.07446,248.08275,251.08887,253.09515,257.10295,257.10777,271.13,270.13,277.15,276.15,eP,eP,283.17,285.18,nP,nP,291.2,PP,PP,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.014,3.01605,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]),e.Kr=v(F(l.j7,1),H,6,15,[6,1,7,8]);},e.Nr=function(t){var A,n;for(n=t.a,A=0;A<t.b.length;A++)n+=t.b[A]*e.Jr[t.c[A]];return n;},e.Or=function(t){var A,n;for(A=new T.Mjb(),n=0;n<t.b.length;n++)T.Ljb(A,(e.Rk(),e.Lk)[t.c[n]]),t.b[n]>1&&T.Kjb(A,t.b[n]);return A.a;},e.Pr=function(t){var A,n;for(n=t.d,A=0;A<t.b.length;A++)n+=t.b[A]*e.Lr[t.c[A]];return n;},e.Qr=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(e.bw(t,1),r=j(l.j7,H,6,191,15,1),n=0;n<t.q;n++)switch(t.C[n]){case 171:r[1]+=5,r[6]+=3,r[7]+=1,r[8]+=1;break;case 172:r[1]+=12,r[6]+=6,r[7]+=4,r[8]+=1;break;case 173:r[1]+=6,r[6]+=4,r[7]+=2,r[8]+=2;break;case 174:r[1]+=5,r[6]+=4,r[7]+=1,r[8]+=3;break;case 175:r[1]+=5,r[6]+=3,r[7]+=1,r[8]+=1,r[16]+=1;break;case 176:r[1]+=8,r[6]+=5,r[7]+=2,r[8]+=2;break;case 177:r[1]+=7,r[6]+=5,r[7]+=1,r[8]+=3;break;case 178:r[1]+=3,r[6]+=2,r[7]+=1,r[8]+=1;break;case 179:r[1]+=7,r[6]+=6,r[7]+=3,r[8]+=1;break;case 181:case 180:r[1]+=11,r[6]+=6,r[7]+=1,r[8]+=1;break;case 182:r[1]+=12,r[6]+=6,r[7]+=2,r[8]+=1;break;case 183:r[1]+=9,r[6]+=5,r[7]+=1,r[8]+=1,r[16]+=1;break;case 184:r[1]+=9,r[6]+=9,r[7]+=1,r[8]+=1;break;case 185:r[1]+=7,r[6]+=5,r[7]+=1,r[8]+=1;break;case 186:r[1]+=5,r[6]+=3,r[7]+=1,r[8]+=2;break;case 187:r[1]+=7,r[6]+=4,r[7]+=1,r[8]+=2;break;case 188:r[1]+=10,r[6]+=11,r[7]+=2,r[8]+=1;break;case 189:r[1]+=9,r[6]+=9,r[7]+=1,r[8]+=2;break;case 190:r[1]+=9,r[6]+=5,r[7]+=1,r[8]+=1;break;case 1:switch(t.A[n]){case 0:case 1:++r[1];break;case 2:++r[151];break;case 3:++r[152];}break;default:++r[t.C[n]];}for(P=0;P<t.q;P++)t.C[P]>=171&&t.C[P]<=190?r[1]+=2-e.vp(t,P):r[1]+=e.op(t,P);for(u=0,Q=1;Q<=190;Q++)r[Q]!=0&&++u;for(this.b=j(l.j7,H,6,u,15,1),this.c=j(l.j7,H,6,u,15,1),u=0,o=0;o<e.Kr.length;o++)r[e.Kr[o]]!=0&&(this.b[u]=r[e.Kr[o]],this.c[u]=e.Kr[o],++u,r[e.Kr[o]]=0);for(;;){for(s="zzz",L=-1,D=1;D<=190;D++)r[D]>0&&T.ejb(s,(e.Rk(),e.Lk)[D])>0&&(s=(e.Rk(),e.Lk)[D],L=D);if(L==-1)break;this.b[u]=r[L],this.c[u]=L,++u,r[L]=0;}for(this.a=0,this.d=0,A=0;A<t.f;A++)t.C[A]!=1&&t.A[A]!=0&&(D=t.C[A],i=t.A[A],this.a+=e.Dr(D,i)-e.Jr[D],this.d+=e.Dr(D,i)-e.Lr[D]);},N(153,1,{153:1}),U.mb=function(t){var A;if(t===this)return  true;if(!Kt(t,153))return  false;for(A=0;A<this.b.length;A++)if(this.b[A]!=t.b[A])return  false;return  true;},U.a=0,U.d=0,l.Q7=O(153),e.Sr=function(t){switch(t){case 6:return 1;case 53:return 2;case 33:return 3;case 34:return 4;case 35:return 5;case 15:return 6;case 16:return 7;case 17:return 8;case 7:return 9;case 8:return 10;case 9:return 11;}return 0;},e.Tr=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z;for(e.bw(t,7),Z=0,b=j(l.Feb,ht,6,t.f,16,1),z=j(l.j7,H,6,t.f,15,1),i=0;i<t.g;i++)n=t.D[0][i],P=t.D[1][i],D=t.s[n],u=t.s[P],D!=0&&u!=0&&D<0^u<0&&(b[n]=true,b[P]=true),z[n]+=e.fm(t,i),z[P]+=e.fm(t,i);for(o=0;o<t.f;o++)if(Z+=t.s[o],t.s[o]==1){if(t.C[o]==7&&!b[o]){if(z[o]<=3)Z-=1,t.s[o]=0,t.T=0,t.j[o]!=t.e[o]&&(e.ql(t,e.Zo(t,o,t.e[o]-1)),e.bw(t,7));else if(o<t.f&&e.ot(t.p,o)){for(k=j(l.Feb,ht,6,t.f,16,1),M=j(l.Feb,ht,6,t.g,16,1),e.Io(t,o,true,k,M),L=0;L<t.f;L++)if(k[L]&&t.C[L]==7&&t.s[L]==0&&z[L]==2&&e.Wr(t,M,o,L)){Z-=1;break;}}}}else if(t.s[o]<0&&(t.C[o]==6||t.C[o]==7||t.C[o]==8||t.C[o]==16))if(b[o]){for(S=j(l.j7,H,6,16,15,1),S[0]=o,I=1,V=0;V<I;){for(r=0;r<t.r;r++)if(A=-1,t.D[0][r]==S[V]?A=t.D[1][r]:t.D[1][r]==S[V]&&(A=t.D[0][r]),A!=-1&&t.s[A]!=0){for(s=false,g=0;g<I;g++)if(A==S[g]){s=true;break;}s||(I==S.length&&(Q=j(l.j7,H,6,2*S.length,15,1),T.$jb(S,0,Q,0,S.length),S=Q),S[I++]=A);}++V;}for(h=0,B=0;B<I;B++)h+=e.Bl(t,S[B]);if(h<0){for(p=-1,q=99,c=0;c<I;c++)e.Bl(t,S[c])<0&&q>e.Sr(e.Tl(t,S[c]))&&(q=e.Sr(e.Tl(t,S[c])),p=S[c]);p!=-1&&(Z-=t.s[p],t.s[p]=0,t.T=0);}}else Z-=t.s[o],t.s[o]=0,t.T=0;return Z>0&&(Z=e.Ur(t,Z)),Z;},e.Ur=function(t,A){var n,P,r,i;if(A>0&&(A=e.Xr(t,A,9)),A>0&&(A=e.Xr(t,A,17)),A>0&&(A=e.Xr(t,A,35)),A>0&&(A=e.Xr(t,A,53)),A>0){for(n=0;n<t.f;n++)if(t.s[n]>0){for(r=false,i=0;i<t.j[n];i++)if(P=t.i[n][i],t.s[P]==0&&e.Hm(t,P)&&e.op(t,P)>0){if(t.s[P]=-1,t.T=0,--A,A==0)return 0;r=true;break;}if(r)continue;}}return A>0&&(A=e.Vr(t,A,8,16)),A>0&&(A=e.Vr(t,A,8,15)),A>0&&(A=e.Vr(t,A,8,6)),A>0&&(A=e.Vr(t,A,7,16)),A;},e.Vr=function(t,A,n,P){var r,i,D,u,o,Q,L,s;for(r=0;r<t.f;r++)if(t.C[r]==n&&t.s[r]==0&&e.op(t,r)>0)for(u=false,Q=0;Q<t.j[r];Q++){if(t.n[r][Q]==1&&(i=t.i[r][Q],t.C[i]==P)){for(s=false,o=false,L=0;L<t.j[i];L++){if(D=t.i[i][L],t.s[D]<0){o=true;break;}D!=r&&t.C[D]==8&&t.n[i][L]==2&&(s=true);}if(!o&&s){if(t.s[r]=-1,t.T=0,--A,A==0)return 0;u=true;}}if(u)break;}return A;},e.Wr=function(t,A,n,P){var r,i,D,u,o,Q,L,s;for(s=j(l.j7,H,6,t.g,15,1),o=j(l.Feb,ht,6,t.g,16,1),D=0;D<t.g;D++)s[D]=t.H[D],A[D]&&(o[D]=true,t.H[D]=1,t.T=0);for(r=0;r<t.f;r++)if(r!=P&&t.o[r]==0&&o[t.k[P][0]])for(L=0;L<t.j[r];L++)u=t.k[r][L],t.H[u]=1,t.T=0,o[u]=false;for(Q=0;Q<t.j[n];Q++)u=t.k[n][Q],t.H[u]=1,t.T=0,o[u]=false;if(e.Ge(new e.Ve(t),o,false))return t.s[n]=0,t.T=0,t.j[P]!=t.e[P]&&e.ql(t,e.Zo(t,P,t.e[P]-1)),e.bw(t,7),true;for(i=0;i<t.g;i++)e.Mn(t,i,s[i]);return  false;},e.Xr=function(t,A,n){var P;for(P=0;P<t.f;P++)if(t.C[P]==n&&t.s[P]==0&&t.j[P]==0&&(t.s[P]=-1,t.T=0,--A,A==0))return 0;return A;},e.Yr=function(t,A){var n,P;for(P=l0.Nmb(t.b,A),n=w.aAb(P).length;n<10;n++)T.Ojb(t.a,32);T.Sjb(t.a,P);},e.Zr=function(t,A){var n,P,r;if(A<0||A>999){T.Sjb(t.a,"  ?");return;}for(n=false,P=0;P<3;P++)r=A/100|0,r==0?P==2||n?T.Ojb(t.a,48):T.Ojb(t.a,32):(T.Ojb(t.a,48+r&n1),n=true),A=10*(A%100);},e.$r=function(t,A){Ct.Cgb(A,t.a.a);},e._r=function(t){e.as.call(this,t);},e.as=function(t){e.cs.call(this,t,new T.Vjb());},e.bs=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt,b1,u1,Ht,gt,w1,x1,L1;for(this.b=new l0.Smb((bt=rP,bt)),It=(T.Zjb(),k1),e.bw(t,15),ct=true,r=0;r<t.f;r++)if((t.u[r]&3)!=0&&(t.u[r]&3)!=3&&(t.u[r]&Q1)>>18!=1){ct=false;break;}if(Et=-1,ct){for(G=j(l.j7,H,6,32,15,1),qt=0,i=0;i<t.f;i++)if((t.u[i]&3)!=0&&(t.u[i]&3)!=3&&(t.u[i]&Q1)>>18==1){Qt=(t.u[i]&Q1)>>18!=1&&(t.u[i]&Q1)>>18!=2?-1:(t.u[i]&F0)>>20,++G[Qt],qt<G[Qt]&&(qt=G[Qt],Et=Qt);break;}}for(this.a=n||new T.Ujb(),Nt=t.P!=null?t.P:"",T.Sjb(this.a,Nt+It),T.Sjb(this.a,"Actelion Java MolfileCreator 1.0"+It+It),e.Zr(this,t.q),e.Zr(this,t.r),T.Sjb(this.a,"  0  0"),e.Zr(this,ct?0:1),T.Sjb(this.a,"  0  0  0  0  0999 V2000"+It),ut=t.q==1,Q=1;Q<t.q;Q++)if(t.J[Q].a!=t.J[0].a||t.J[Q].b!=t.J[0].b||t.J[Q].c!=t.J[0].c){ut=true;break;}if(it=1,ut)if(A!=0)it=A;else if(p=e.Wl(t,t.q,t.r,(e.Rk(),e.Qk)),p!=0)(p<1||p>3)&&(it=1.5/p);else {for(Zt=U0,i=1;i<t.q;i++)for(o=0;o<i;o++)z=t.J[o].a-t.J[i].a,tt=t.J[o].b-t.J[i].b,ot=t.J[o].c-t.J[i].c,Z=z*z+tt*tt+ot*ot,Zt>Z&&(Zt=Z);it=3/Zt;}for(L=0;L<t.q;L++)ut?(e.Yr(this,it*t.J[L].a),e.Yr(this,it*-t.J[L].b),e.Yr(this,it*-t.J[L].c)):T.Sjb(this.a,"    0.0000    0.0000    0.0000"),(t.v==null?null:t.v[L])!=null?T.Sjb(this.a," L  "):pt(J(t.B[L],1),0)?T.Sjb(this.a," A  "):t.C[L]>=129&&t.C[L]<=144||t.C[L]==154?T.Sjb(this.a," R# "):(B=(e.Rk(),e.Lk)[t.C[L]],T.Sjb(this.a," "+B),w.aAb(B).length==1?T.Sjb(this.a,"  "):w.aAb(B).length==2&&T.Sjb(this.a," ")),T.Sjb(this.a," 0  0  0"),Dt=J(X0,t.B[L]),et(Dt,0)==0?T.Sjb(this.a,"  0"):et(Dt,384)==0?T.Sjb(this.a,"  3"):et(Dt,128)==0?T.Sjb(this.a,"  2"):et(Dt,1792)==0?T.Sjb(this.a,"  1"):et(Dt,1664)==0&&T.Sjb(this.a,"  2"),T.Sjb(this.a,pt(J(t.B[L],L0),0)?"  1":"  0"),L1=((t.u[L]&cA)>>>27)-1,L1==-1?T.Sjb(this.a,"  0"):L1==0?T.Sjb(this.a," 15"):e.Zr(this,L1),T.Sjb(this.a,"  0  0  0"),e.Zr(this,d.Math.abs(t.w[L])),T.Sjb(this.a,"  0  0"+It);for(q=0;q<t.r;q++){switch(t.H[q]){case 1:jt=1,gt=0;break;case 2:jt=2,gt=0;break;case 4:jt=3,gt=0;break;case 129:jt=1,gt=6;break;case 257:jt=1,gt=1;break;case 386:jt=2,gt=3;break;case 8:jt=4,gt=0;break;case 16:jt=8,gt=0;break;default:jt=1,gt=0;}ct&&(gt==1||gt==6)&&(i=t.D[0][q],(t.u[i]&Q1)>>18==2?gt=0:(t.u[i]&Q1)>>18==1&&((t.u[i]&Q1)>>18!=1&&(t.u[i]&Q1)>>18!=2?-1:(t.u[i]&F0)>>20)!=Et&&(gt=4)),S=t.G[q]&31,S!=0&&(S==8?jt=4:S==3?jt=5:S==9?jt=6:S==10?jt=7:jt=8),u1=t.G[q]&384,x1=u1==0?0:u1==256?1:2,e.Zr(this,1+t.D[0][q]),e.Zr(this,1+t.D[1][q]),e.Zr(this,jt),e.Zr(this,gt),T.Sjb(this.a,"  0"),e.Zr(this,x1),T.Sjb(this.a,"  0"+It);}for(Bt=0,s=0;s<t.q;s++)t.s[s]!=0&&++Bt;if(Bt!=0)for(I=0,i=0;i<t.q;i++)t.s[i]!=0&&(I==0&&(T.Sjb(this.a,"M  CHG"),e.Zr(this,d.Math.min(8,Bt))),T.Sjb(this.a," "),e.Zr(this,i+1),V=t.s[i],V<0?(T.Sjb(this.a,"  -"),V=-V):T.Sjb(this.a,"   "),T.Ojb(this.a,48+V&n1),--Bt,(++I==8||Bt==0)&&(I=0,T.Sjb(this.a,It)));for(Bt=0,h=0;h<t.q;h++)t.A[h]==0||++Bt;if(Bt!=0)for(I=0,i=0;i<t.q;i++)t.A[i]!=0&&(I==0&&(T.Sjb(this.a,"M  ISO"),e.Zr(this,d.Math.min(8,Bt))),T.Sjb(this.a," "),e.Zr(this,i+1),T.Sjb(this.a," "),e.Zr(this,t.A[i]),--Bt,(++I==8||Bt==0)&&(I=0,T.Sjb(this.a,It)));for(Bt=0,b=0;b<t.q;b++)(t.u[b]&48)!=0&&++Bt;if(Bt!=0){for(I=0,i=0;i<t.q;i++)if((t.u[i]&48)!=0){switch(I==0&&(T.Sjb(this.a,"M  RAD"),e.Zr(this,d.Math.min(8,Bt))),T.Sjb(this.a," "),e.Zr(this,i+1),t.u[i]&48){case 16:T.Sjb(this.a,"   1");break;case 32:T.Sjb(this.a,"   2");break;case 48:T.Sjb(this.a,"   3");}--Bt,(++I==8||Bt==0)&&(I=0,T.Sjb(this.a,It));}}for(Bt=0,c=0;c<t.q;c++)(t.C[c]>=129&&t.C[c]<=144||t.C[c]==154)&&++Bt;if(Bt!=0)for(I=0,i=0;i<t.q;i++)M=t.C[i],(M>=129&&M<=144||M==154)&&(I==0&&(T.Sjb(this.a,"M  RGP"),e.Zr(this,d.Math.min(8,Bt))),T.Sjb(this.a," "),e.Zr(this,i+1),T.Sjb(this.a," "),e.Zr(this,M==154?0:M>=142?M-141:M-125),--Bt,(++I==8||Bt==0)&&(I=0,T.Sjb(this.a,It)));if(t.K){for(Bt=0,i=0;i<t.q;i++)pt(J(t.B[i],120),0)&&++Bt;if(Bt!=0)for(I=0,o=0;o<t.q;o++)b1=J(t.B[o],120),et(b1,0)!=0&&(I==0&&(T.Sjb(this.a,"M  RBC"),e.Zr(this,d.Math.min(8,Bt))),T.Sjb(this.a," "),e.Zr(this,o+1),et(b1,112)==0?T.Sjb(this.a,"  -1"):et(b1,8)==0?T.Sjb(this.a,"   1"):et(b1,104)==0?T.Sjb(this.a,"   2"):et(b1,88)==0?T.Sjb(this.a,"   3"):et(b1,56)==0&&T.Sjb(this.a,"   4"),--Bt,(++I==8||Bt==0)&&(I=0,T.Sjb(this.a,It)));for(g=0;g<t.q;g++)if(k=t.v==null?null:t.v[g],k!=null){for(T.Sjb(this.a,"M  ALS "),e.Zr(this,g+1),e.Zr(this,k.length),T.Sjb(this.a,pt(J(t.B[g],1),0)?" T ":" F "),lt=0;lt<k.length;lt++)switch(yt=(e.Rk(),e.Lk)[k[lt]],w.aAb(yt).length){case 1:T.Sjb(this.a,yt+"   ");break;case 2:T.Sjb(this.a,yt+"  ");break;case 3:T.Sjb(this.a,yt+" ");break;default:T.Sjb(this.a,"   ?");}T.Sjb(this.a,It);}for(Bt=0,D=0;D<t.q;D++)pt(J(t.B[D],6144),0)&&++Bt;if(Bt!=0)for(I=0,o=0;o<t.q;o++)w1=J(t.B[o],6144),et(w1,0)!=0&&(I==0&&(T.Sjb(this.a,"M  SUB"),e.Zr(this,d.Math.min(8,Bt))),T.Sjb(this.a," "),e.Zr(this,o+1),pt(J(w1,Z1),0)?T.Sjb(this.a,"   "+(t.e[o]+1)):T.Sjb(this.a,"  -2"),--Bt,(++I==8||Bt==0)&&(I=0,T.Sjb(this.a,It)));}for(Bt=0,u=0;u<t.q;u++)(t.t==null||t.t[u]==null?null:T.ijb((xt=t.t[u],S1.Mmb(),xt)))!=null&&++Bt;if(Bt!=0)for(Ht=0,P=0;P<t.q;P++)yt=t.t==null||t.t[P]==null?null:T.ijb((wt=t.t[P],S1.Mmb(),wt)),yt!=null&&(++Ht,T.Sjb(this.a,"M  STY  1 "),e.Zr(this,Ht),T.Sjb(this.a," DAT"),T.Sjb(this.a,It),T.Sjb(this.a,"M  SLB  1 "),e.Zr(this,Ht),T.Sjb(this.a," "),e.Zr(this,Ht),T.Sjb(this.a,It),T.Sjb(this.a,"M  SAL "),e.Zr(this,Ht),T.Sjb(this.a,"  1 "),e.Zr(this,P+1),T.Sjb(this.a,It),T.Sjb(this.a,"M  SDT "),e.Zr(this,Ht),T.Sjb(this.a," "),T.Sjb(this.a,iP),T.Sjb(this.a,It),T.Sjb(this.a,"M  SDD "),e.Zr(this,Ht),T.Sjb(this.a," "),e.Yr(this,t.J[P].a),e.Yr(this,t.J[P].b),T.Sjb(this.a,"    DA    ALL  1       5"),T.Sjb(this.a,It),T.Sjb(this.a,"M  SED "),e.Zr(this,Ht),T.Sjb(this.a," "),T.Sjb(this.a,yt),T.Sjb(this.a,It));T.Sjb(this.a,FA+It);},e.cs=function(t,A){e.bs.call(this,t,0,A);},N(112,1,{},e._r,e.bs),l.S7=O(112),e.ds=function(t,A,n,P,r,i){var D,u,o,Q;switch(Q=1,u=false,r){case 1:Q=257;break;case 3:Q=386;break;case 4:Q=257,u=true;break;case 6:Q=129;break;default:switch(P){case 1:Q=1;break;case 2:Q=2;break;case 3:Q=4;break;case 4:Q=8;break;case 9:Q=16;}}if(D=e.Wk(t.j,A,n,Q),o=0,u&&e.on(t.j,A,1,-1),P>4)switch(P){case 5:o|=3;break;case 6:o|=9;break;case 7:o|=10;break;case 8:Q!=16&&(o|=127);}return i==1&&(o|=256),i==2&&(o|=128),o!=0&&e.Ln(t.j,D,o,true),D;},e.es=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(D=j(l.j7,H,6,t.j.q,15,1),n=0;n<t.j.q;n++)D[n]=-e.Bl(t.j,n);for(e.Ge(new e.Ve(t.j),null,true),P=0;P<t.j.q;P++)D[P]+=e.Bl(t.j,P);for(A=0;A<t.j.q;A++)if(D[A]!=0)for(u=-D[A],r=0;r<t.j.r;r++)for(Q=0;Q<2;Q++)u>0&&e.im(t.j,r)==16&&e.Zl(t.j,1-Q,r)==A&&(s=e.Zl(t.j,Q,r),e.Lm(t.j,s)&&(L=e.gs(t,s),i=e.Bl(t.j,s),i<L&&(o=d.Math.min(u,L-i),e.jn(t.j,s,i+o),u-=o)));},e.fs=function(t,A){return t.j=null,e.As(t,new Ct.ogb(new Ct.zgb(A)))?t.j:null;},e.gs=function(t,A){var n,P;return n=e.Tl(t.j,A),P=n<(e.Rk(),e.Nk).length?e.Nk[n]:null,P==null?0:P[P.length-1];},e.hs=function(t,A){var n;return n=t.b?y.opb(t.b,T.Aib(A)):null,n?n.a:A-1;},e.is=function(t,A){var n;return n=t.c?y.opb(t.c,T.Aib(A)):null,n?n.a:A-1;},e.js=function(t,A){var n,P;if(A!=null)for(e.bw(t.j,1),n=0;n<t.j.f;n++)A[n]!=0&&(P=e.mm(t.j,n,e.vp(t.j,n)),A[n]==15?P>=0&&e.gn(t.j,n,0):A[n]!=e.om(t.j,n)&&e.gn(t.j,n,A[n]-P));},e.ks=function(t,A){var n,P;for(n=A;n<w.aAb(t).length;n++){if(P=(w.Xzb(n,w.aAb(t).length),w.aAb(t).charCodeAt(n)),P==32||P==9)return  -1;if(P==61)return n;}return  -1;},e.ls=function(t,A){var n;if(A==-1)return  -1;for(n=A+1;n<w.aAb(t).length;n++)if(w.Xzb(n,w.aAb(t).length),w.aAb(t).charCodeAt(n)!=32&&(w.Xzb(n,w.aAb(t).length),w.aAb(t).charCodeAt(n)!=9))return n;return  -1;},e.ms=function(t,A){var n,P,r,i;for(n=false,r=false,P=A;P<w.aAb(t).length;P++)if(i=(w.Xzb(P,w.aAb(t).length),w.aAb(t).charCodeAt(P)),i==40?n=true:i==41?n=false:i==34&&(r=!r),!n&&!r&&(i==32||i==9))return P;return  -1;},e.ns=function(t){return w.aAb(t).indexOf("ATOMS=(")!=-1?Ee:w.aAb(t).indexOf("BONDS=(")!=-1?"BONDS":null;},e.os=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot;for(g=0,B=(u=e.ms(A,g),u==-1?w.aAb(A).length:u),P=T.Whb((w.Wzb(g,B,w.aAb(A).length),w.aAb(A).substr(g,B-g)),P1,St),g=e.ls(A,B),B=(o=e.ms(A,g),o==-1?w.aAb(A).length:o),M=(w.Wzb(g,B,w.aAb(A).length),w.aAb(A).substr(g,B-g)),I=null,r=false,k=e.ws(A),k!=0&&(I=e.ps(A),k<0&&(r=true),B=d.Math.abs(k)),g=e.ls(A,B),B=(Q=e.ms(A,g),Q==-1?w.aAb(A).length:Q),z=N0((w.Wzb(g,B,w.aAb(A).length),w.aAb(A).substr(g,B-g))),g=e.ls(A,B),B=(L=e.ms(A,g),L==-1?w.aAb(A).length:L),tt=N0((w.Wzb(g,B,w.aAb(A).length),w.aAb(A).substr(g,B-g))),g=e.ls(A,B),B=(s=e.ms(A,g),s==-1?w.aAb(A).length:s),ot=N0((w.Wzb(g,B,w.aAb(A).length),w.aAb(A).substr(g,B-g))),g=e.ls(A,B),B=(h=e.ms(A,g),h==-1?w.aAb(A).length:h),q=T.Whb((w.Wzb(g,B,w.aAb(A).length),w.aAb(A).substr(g,B-g)),P1,St),n=e.Tk(t.j,z,-tt,-ot),n+1!=P&&(!t.b&&(t.b=new y.fwb()),y.Zvb(t.b,T.Aib(P),T.Aib(n))),I!=null&&(e.Cn(t.j,n,6),e.qn(t.j,n,I,r)),q!=0&&e.rn(t.j,n,q,false),T.kjb(M,"A")||T.kjb(M,"*")?e.vn(t.j,n,1,true):T.kjb(M,"Q")?(p=j(l.j7,H,6,1,15,1),p[0]=6,e.qn(t.j,n,p,true)):e.Cn(t.j,n,e.po(M,67));(g=e.ls(A,B))!=-1;)if(B=(D=e.ms(A,g),D==-1?w.aAb(A).length:D),S=(w.Wzb(g,B,w.aAb(A).length),w.aAb(A).substr(g,B-g)),c=T.rjb(S,Y1(61)),b=(w.Wzb(0,c,w.aAb(S).length),w.aAb(S).substr(0,c)),Z=T.Whb((w.Xzb(c+1,w.aAb(S).length+1),w.aAb(S).substr(c+1)),P1,St),T.kjb(b,"CHG"))e.jn(t.j,n,Z);else if(T.kjb(b,"RAD"))switch(Z){case 1:e.wn(t.j,n,16);break;case 2:e.wn(t.j,n,32);break;case 3:e.wn(t.j,n,48);}else if(!T.kjb(b,"CFG")){if(T.kjb(b,"MASS"))e.tn(t.j,n,Z);else if(T.kjb(b,"VAL"))e.gn(t.j,n,Z==-1?0:Z==0?-1:Z);else if(T.kjb(b,"HCOUNT"))switch(Z){case 0:break;case  -1:e.vn(t.j,n,1792,true);break;case 1:e.vn(t.j,n,128,true);break;case 2:e.vn(t.j,n,384,true);break;default:e.vn(t.j,n,896,true);}else if(T.kjb(b,"SUBST")){if(Z==-1)e.vn(t.j,n,j1,true);else if(Z>0){for(V=0,i=0;i<t.j.r;i++)(e.Zl(t.j,0,i)==n||e.Zl(t.j,1,i)==n)&&++V;Z>V&&e.vn(t.j,n,Z1,true);}}else if(T.kjb(b,"RBCNT"))switch(Z){case 3:case  -1:e.vn(t.j,n,112,true);break;case 1:e.vn(t.j,n,8,true);break;case 2:e.vn(t.j,n,104,true);break;case 4:e.vn(t.j,n,56,true);}}},e.ps=function(t){var A,n,P,r,i,D,u,o;if(u=null,n=w.aAb(t).indexOf("["),P=w.aAb(t).indexOf("]",n),n>=0&&P>0){for(A=j(l.j7,H,6,16,15,1),o=(w.Wzb(n+1,P,w.aAb(t).length),w.aAb(t).substr(n+1,P-(n+1))),r=0,D=true;D&&r<16;)n=w.aAb(o).indexOf(","),i=null,n==-1?(i=o,D=false):(i=(w.Wzb(0,n,w.aAb(o).length),w.aAb(o).substr(0,n)),o=(w.Xzb(n+1,w.aAb(o).length+1),w.aAb(o).substr(n+1))),A[r++]=e.po(i,1);u=j(l.j7,H,6,r,15,1),T.$jb(A,0,u,0,r);}return u;},e.qs=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p;for(c=0,g=(o=e.ms(A,c),o==-1?w.aAb(A).length:o),i=T.Whb((w.Wzb(c,g,w.aAb(A).length),w.aAb(A).substr(c,g-c)),P1,St),c=e.ls(A,g),g=(Q=e.ms(A,c),Q==-1?w.aAb(A).length:Q),D=T.Whb((w.Wzb(c,g,w.aAb(A).length),w.aAb(A).substr(c,g-c)),P1,St),c=e.ls(A,g),g=(L=e.ms(A,c),L==-1?w.aAb(A).length:L),n=e.hs(t,T.Whb((w.Wzb(c,g,w.aAb(A).length),w.aAb(A).substr(c,g-c)),P1,St)),c=e.ls(A,g),g=(s=e.ms(A,c),s==-1?w.aAb(A).length:s),P=e.hs(t,T.Whb((w.Wzb(c,g,w.aAb(A).length),w.aAb(A).substr(c,g-c)),P1,St)),k=0,M=0;(c=e.ls(A,g))!=-1;)if(g=(u=e.ms(A,c),u==-1?w.aAb(A).length:u),B=(w.Wzb(c,g,w.aAb(A).length),w.aAb(A).substr(c,g-c)),b=T.rjb(B,Y1(61)),h=(w.Wzb(0,b,w.aAb(B).length),w.aAb(B).substr(0,b)),p=T.Whb((w.Xzb(b+1,w.aAb(B).length+1),w.aAb(B).substr(b+1)),P1,St),T.kjb(h,"CFG"))switch(p){case 1:k=1;break;case 2:k=D==2?3:4;break;case 3:k=6;}else T.kjb(h,"TOPO")&&(M=p);r=e.ds(t,n,P,D,k,M),r+1!=i&&(!t.c&&(t.c=new y.fwb()),y.Zvb(t.c,T.Aib(i),T.Aib(r)));},e.rs=function(t,A){var n,P,r,i,D,u;if(u=e.ns(A),u!=null){if(D=e.ts(A,u),T.kjb(w.aAb(A).substr(0,13),"MDLV30/STEABS")){if(T.kjb(u,Ee))for(i=0;i<D.length;i++)e.on(t.j,e.hs(t,D[i]),0,-1);else for(r=0;r<D.length;r++)e.Gn(t.j,e.is(t,D[r]),0,-1);}else if(T.kjb(w.aAb(A).substr(0,13),"MDLV30/STERAC")){if(P=T.Whb(T.Bjb(A,13,e.ms(A,13)),P1,St),T.kjb(u,Ee))for(i=0;i<D.length;i++)e.on(t.j,e.hs(t,D[i]),1,P-1);else for(r=0;r<D.length;r++)e.Gn(t.j,e.is(t,D[r]),1,P-1);}else if(T.kjb(w.aAb(A).substr(0,13),"MDLV30/STEREL")){if(P=T.Whb(T.Bjb(A,13,e.ms(A,13)),P1,St),T.kjb(u,Ee))for(i=0;i<D.length;i++)e.on(t.j,e.hs(t,D[i]),2,P-1);else for(r=0;r<D.length;r++)e.Gn(t.j,e.is(t,D[r]),2,P-1);}else if(T.kjb(w.aAb(A).substr(0,13),"MDLV30/HILITE"))if(T.kjb(u,Ee))for(r=0;r<D.length;r++)e.kn(t.j,e.hs(t,D[r]),448);else for(r=0;r<D.length;r++)n=e.is(t,D[r]),e.kn(t.j,e.Zl(t.j,0,n),448),e.kn(t.j,e.Zl(t.j,1,n),448);}},e.ss=function(t,A){var n,P,r;t.j||T.kjb(w.aAb(A).substr(0,6),"COUNTS")&&(n=e.ls(A,e.ms(A,7)),P=T.Whb(T.Bjb(A,7,e.ms(A,7)),P1,St),r=T.Whb(T.Bjb(A,n,e.ms(A,n)),P1,St),t.j=new e.uw(P,r));},e.ts=function(t,A){var n,P,r,i,D,u;for(i=w.aAb(t).indexOf(A+"=(")+w.aAb(A).length+2,D=T.sjb(t,Y1(41),i),r=e.ms(t,i),n=T.Whb((w.Wzb(i,r,w.aAb(t).length),w.aAb(t).substr(i,r-i)),P1,St),u=j(l.j7,H,6,n,15,1),P=0;P<n;P++)i=e.ls(t,r),r=e.ms(t,i),(r==-1||r>D)&&(r=D),u[P]=T.Whb((w.Wzb(i,r,w.aAb(t).length),w.aAb(t).substr(i,r-i)),P1,St);return u;},e.us=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p;if(h=0,b=(i=e.ms(A,h),i==-1?w.aAb(A).length:i),p=(w.Wzb(h,b,w.aAb(A).length),w.aAb(A).substr(h,b-h)),T.kjb("DEFAULT",p)&&(h=e.ls(A,b),b=(D=e.ms(A,h),D==-1?w.aAb(A).length:D),p=(w.Wzb(h,b,w.aAb(A).length),w.aAb(A).substr(h,b-h)),T.kjb(w.aAb(p).substr(0,5),"CLASS")&&(h=e.ls(A,b),b=(u=e.ms(A,h),u==-1?w.aAb(A).length:u),p=(w.Wzb(h,b,w.aAb(A).length),w.aAb(A).substr(h,b-h)),T.kjb("-",p)&&(h=e.ls(A,b),b=(o=e.ms(A,h),o==-1?w.aAb(A).length:o),p=(w.Wzb(h,b,w.aAb(A).length),w.aAb(A).substr(h,b-h))))),T.Whb(p,P1,St),h=e.ls(A,b),b=(Q=e.ms(A,h),Q==-1?w.aAb(A).length:Q),M=(w.Wzb(h,b,w.aAb(A).length),w.aAb(A).substr(h,b-h)),!!T.kjb("DAT",M)){for(h=e.ls(A,b),b=(r=e.ms(A,h),r==-1?w.aAb(A).length:r),T.Whb((w.Wzb(h,b,w.aAb(A).length),w.aAb(A).substr(h,b-h)),P1,St),k=new y.fwb(),h=e.ls(A,b);h!=-1;)h=e.zs(k,A,h);P=(g=Ee,y.fnb(y.Uvb(k,g))),s=(B="FIELDNAME",y.fnb(y.Uvb(k,B))),L=(c="FIELDDATA",y.fnb(y.Uvb(k,c))),!(!T.kjb(iP,s)||P==null||L==null)&&(n=T.yjb(P," "),n.length==2&&e.mn(t.j,T.Whb(n[1],P1,St)-1,L));}},e.vs=function(t){var A,n,P,r,i,D,u,o,Q;for(o=j(l.j7,H,6,t.j.q,15,1),P=0;P<t.j.r;P++)if(e.im(t.j,P)==8)for(D=0;D<2;D++)o[e.Zl(t.j,D,P)]=1;for(r=0;r<t.j.r;r++)for(Q=e.fm(t.j,r),i=0;i<2;i++)o[e.Zl(t.j,i,r)]+=Q;for(n=0;n<t.j.r;n++)if(e.fm(t.j,n)==1){for(i=0;i<2;i++)if(u=e.Zl(t.j,1-i,n),e.Lm(t.j,u)&&(A=e.Zl(t.j,i,n),e.Hm(t.j,A)&&o[A]>e.om(t.j,A))){e.Mn(t.j,n,16);continue;}}},e.ws=function(t){var A,n;if(w.aAb(t).indexOf("[")>=0){if(A=w.aAb(t).indexOf(" NOT["),n=w.aAb(t).indexOf("]",A),A>=0&&n>0)return -(n+1);if(A=w.aAb(t).indexOf(" ["),n=w.aAb(t).indexOf("]",A),A>=0&&n>0)return n+1;if(A=w.aAb(t).indexOf(" 'NOT["),n=w.aAb(t).indexOf("]'",A),A>=0&&n>0)return -(n+2);if(A=w.aAb(t).indexOf(" '["),n=w.aAb(t).indexOf("]'",A),A>=0&&n>0)return n+2;T.Zjb();}return 0;},e.xs=function(t,A,n){return t.j=A,e.As(t,n);},e.ys=function(t){return w.aAb(t).length==0?0:T.Whb(t,P1,St);},e.zs=function(t,A,n){var P,r,i,D,u,o,Q;return i=e.ks(A,n),i==-1?-1:(r=(P=e.ms(A,i+1),P==-1?w.aAb(A).length:P),D=(w.Wzb(n,i,w.aAb(A).length),w.aAb(A).substr(n,i-n)),Q=(w.Wzb(i+1,r,w.aAb(A).length),w.aAb(A).substr(i+1,r-(i+1))),(T.kjb(w.aAb(Q).substr(0,1),"(")&&(o=w.aAb(")").length,T.kjb(w.aAb(Q).substr(w.aAb(Q).length-o,o),")"))||T.kjb(w.aAb(Q).substr(0,1),'"')&&(u=w.aAb('"').length,T.kjb(w.aAb(Q).substr(w.aAb(Q).length-u,u),'"')))&&(Q=T.Bjb(Q,1,w.aAb(Q).length-1)),y.Zvb(t,D,Q),e.ls(A,r));},e.As=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt,b1,u1,Ht,gt,w1,x1;u1=null,c=null;try{if(t.f=null,t.j&&(e._v(t.j),e.On(t.j,!1)),Qt=z=Ct.ngb(A),Qt==null||(z=Ct.ngb(A))==null||(z=Ct.ngb(A))==null||(t.k=w.aAb(z).indexOf("From CSD data. Using bond type 'Any'")!=-1,t.e=w.aAb(z).indexOf("From CSD data.")!=-1,(z=Ct.ngb(A))==null))return !1;t.g=!1,t.d=t.a;try{ut=T.Whb(T.Cjb((w.Wzb(0,3,w.aAb(z).length),w.aAb(z).substr(0,3))),P1,St),Dt=T.Whb(T.Cjb((w.Wzb(3,6,w.aAb(z).length),w.aAb(z).substr(3,3))),P1,St),lt=e.ys(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3)))),t.d=t.d|e.ys(T.Cjb((w.Wzb(12,15,w.aAb(z).length),w.aAb(z).substr(12,3))))==1,t.g=w.aAb(z).length>=39&&T.kjb(w.aAb(z).substr(34,w.aAb("V3000").length),"V3000");}catch(L1){if(L1=M1(L1),Kt(L1,19))return !1;throw Ut(L1);}if(t.g)return bt=e.Bs(t,A),e.Un(t.j,Qt),bt;if(!t.j&&(t.j=new e.uw(ut,Dt)),e.Un(t.j,Qt),t.d||(t.j.L=!0),ut==0){for(;z!=null&&!(T.kjb(z,FA)||T.kjb(z,Qn)||T.kjb((w.Xzb(1,w.aAb(z).length+1),w.aAb(z).substr(1)),"$"));)z=Ct.ngb(A);return !0;}for(p=0;p<ut;p++){if((z=Ct.ngb(A))==null)return !1;switch(gt=N0(T.Cjb((w.Wzb(0,10,w.aAb(z).length),w.aAb(z).substr(0,10)))),w1=N0(T.Cjb((w.Wzb(10,20,w.aAb(z).length),w.aAb(z).substr(10,10)))),x1=N0(T.Cjb((w.Wzb(20,30,w.aAb(z).length),w.aAb(z).substr(20,10)))),P=e.Tk(t.j,gt,-w1,-x1),Z=T.Cjb((w.Wzb(31,34,w.aAb(z).length),w.aAb(z).substr(31,3))),T.kjb(Z,"A")||T.kjb(Z,"*")?e.vn(t.j,P,1,!0):T.kjb(Z,"Q")?(tt=j(l.j7,H,6,1,15,1),tt[0]=6,e.qn(t.j,P,tt,!0)):(D=e.po(Z,67),e.Cn(t.j,P,D)),it=e.ys(T.Cjb((w.Wzb(34,36,w.aAb(z).length),w.aAb(z).substr(34,2)))),it!=0&&e.tn(t.j,P,(e.Rk(),e.Pk)[e.Tl(t.j,P)]+it),s=e.ys(T.Cjb((w.Wzb(36,39,w.aAb(z).length),w.aAb(z).substr(36,3)))),s!=0&&(s==4?e.wn(t.j,P,32):e.jn(t.j,P,4-s)),ot=w.aAb(z).length<63?0:e.ys(T.Cjb((w.Wzb(60,63,w.aAb(z).length),w.aAb(z).substr(60,3)))),e.rn(t.j,P,ot,!1),k=w.aAb(z).length<45?0:e.ys(T.Cjb((w.Wzb(42,45,w.aAb(z).length),w.aAb(z).substr(42,3)))),k){case 0:break;case 1:e.vn(t.j,P,768,!0);break;case 2:e.vn(t.j,P,128,!0);break;case 3:e.vn(t.j,P,384,!0);break;default:e.vn(t.j,P,896,!0);}w.aAb(z).length>=48&&(w.Xzb(47,w.aAb(z).length),w.aAb(z).charCodeAt(47)==49)&&e.vn(t.j,P,L0,!0),b1=w.aAb(z).length<51?0:e.ys(T.Cjb((w.Wzb(48,51,w.aAb(z).length),w.aAb(z).substr(48,3)))),b1!=0&&(u1==null&&(u1=j(l.j7,H,6,ut,15,1)),u1[P]=b1);}for(q=0;q<Dt;q++){if((z=Ct.ngb(A))==null)return !1;r=T.Whb(T.Cjb((w.Wzb(0,3,w.aAb(z).length),w.aAb(z).substr(0,3))),P1,St)-1,i=T.Whb(T.Cjb((w.Wzb(3,6,w.aAb(z).length),w.aAb(z).substr(3,3))),P1,St)-1,Q=T.Whb(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3))),P1,St),Zt=w.aAb(z).length<12?0:e.ys(T.Cjb((w.Wzb(9,12,w.aAb(z).length),w.aAb(z).substr(9,3)))),jt=w.aAb(z).length<18?0:e.ys(T.Cjb((w.Wzb(15,18,w.aAb(z).length),w.aAb(z).substr(15,3)))),Q==8&&(t.k||e.Lm(t.j,r)||e.Lm(t.j,i))&&(Q=9),e.ds(t,r,i,Q,Zt,jt);}for(M=0;M<lt;M++)if((z=Ct.ngb(A))==null)return !1;if((z=Ct.ngb(A))==null)return (t.i&1)!=0&&(t.f=e.lp(t.j)),e.js(t,u1),t.d||e.bw(t.j,15),!0;for(;z!=null&&!(T.kjb(z,FA)||T.kjb(z,Qn));){if(T.kjb(w.aAb(z).substr(0,6),"M  CHG")&&(S=T.Whb(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3))),P1,St),S>0))for(n=10,Ht=14,V=1;V<=S;++V,n+=8,Ht+=8)P=T.Whb(T.Cjb((w.Wzb(n,n+3,w.aAb(z).length),w.aAb(z).substr(n,n+3-n))),P1,St)-1,L=T.Whb(T.Cjb((w.Wzb(Ht,Ht+3,w.aAb(z).length),w.aAb(z).substr(Ht,Ht+3-Ht))),P1,St),e.jn(t.j,P,L);if(T.kjb(w.aAb(z).substr(0,6),"M  ISO")&&(S=T.Whb(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3))),P1,St),S>0))for(n=10,Ht=14,V=1;V<=S;++V,n+=8,Ht+=8)P=T.Whb(T.Cjb((w.Wzb(n,n+3,w.aAb(z).length),w.aAb(z).substr(n,n+3-n))),P1,St)-1,G=T.Whb(T.Cjb((w.Wzb(Ht,Ht+3,w.aAb(z).length),w.aAb(z).substr(Ht,Ht+3-Ht))),P1,St),e.tn(t.j,P,G);if(T.kjb(w.aAb(z).substr(0,6),"M  RAD")&&(S=T.Whb(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3))),P1,St),S>0))for(n=10,Ht=14,V=1;V<=S;++V,n+=8,Ht+=8)switch(P=T.Whb(T.Cjb((w.Wzb(n,n+3,w.aAb(z).length),w.aAb(z).substr(n,n+3-n))),P1,St)-1,wt=T.Whb(T.Cjb((w.Wzb(Ht,Ht+3,w.aAb(z).length),w.aAb(z).substr(Ht,Ht+3-Ht))),P1,St),wt){case 1:e.wn(t.j,P,16);break;case 2:e.wn(t.j,P,32);break;case 3:e.wn(t.j,P,48);}if((T.kjb(w.aAb(z).substr(0,6),"M  RBC")||T.kjb(w.aAb(z).substr(0,6),"M  RBD"))&&(S=T.Whb(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3))),P1,St),S>0))for(n=10,Ht=14,V=1;V<=S;++V,n+=8,Ht+=8)switch(P=T.Whb(T.Cjb((w.Wzb(n,n+3,w.aAb(z).length),w.aAb(z).substr(n,n+3-n))),P1,St)-1,xt=T.Whb(T.Cjb((w.Wzb(Ht,Ht+3,w.aAb(z).length),w.aAb(z).substr(Ht,Ht+3-Ht))),P1,St),xt){case 3:case -1:e.vn(t.j,P,112,!0);break;case 1:e.vn(t.j,P,8,!0);break;case 2:e.vn(t.j,P,104,!0);break;case 4:e.vn(t.j,P,56,!0);}if(T.kjb(w.aAb(z).substr(0,6),"M  ALS")&&(P=T.Whb(T.Cjb((w.Wzb(7,10,w.aAb(z).length),w.aAb(z).substr(7,3))),P1,St)-1,P>=0)){for(ct=T.Whb(T.Cjb((w.Wzb(10,13,w.aAb(z).length),w.aAb(z).substr(10,3))),P1,St),u=(w.Xzb(14,w.aAb(z).length),w.aAb(z).charCodeAt(14)==84),b1=j(l.j7,H,6,ct,15,1),n=16,V=0;V<ct;++V,n+=4)Bt=T.Cjb((w.Wzb(n,n+4,w.aAb(z).length),w.aAb(z).substr(n,n+4-n))),b1[V]=e.po(Bt,1);e.Cn(t.j,P,6),e.qn(t.j,P,b1,u);}if(T.kjb(w.aAb(z).substr(0,6),"M  SUB")&&(S=T.Whb(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3))),P1,St),S>0)){for(n=10,Ht=14,V=1;V<=S;++V,n+=8,Ht+=8)if(P=T.Whb(T.Cjb((w.Wzb(n,n+3,w.aAb(z).length),w.aAb(z).substr(n,n+3-n))),P1,St)-1,Nt=T.Whb(T.Cjb((w.Wzb(Ht,Ht+3,w.aAb(z).length),w.aAb(z).substr(Ht,Ht+3-Ht))),P1,St),Nt==-2)e.vn(t.j,P,j1,!0);else if(Nt>0){for(It=0,o=0;o<t.j.r;o++)(e.Zl(t.j,0,o)==P||e.Zl(t.j,1,o)==P)&&++It;Nt>It&&e.vn(t.j,P,Z1,!0);}}if(T.kjb(w.aAb(z).substr(0,6),"M  RGP")&&(S=T.Whb(T.Cjb((w.Wzb(6,9,w.aAb(z).length),w.aAb(z).substr(6,3))),P1,St),S>0))for(n=10,Ht=14,V=1;V<=S;++V,n+=8,Ht+=8)P=T.Whb(T.Cjb((w.Wzb(n,n+3,w.aAb(z).length),w.aAb(z).substr(n,n+3-n))),P1,St)-1,Et=T.Whb(T.Cjb((w.Wzb(Ht,Ht+3,w.aAb(z).length),w.aAb(z).substr(Ht,Ht+3-Ht))),P1,St),Et>=1&&Et<=20&&e.Cn(t.j,P,e.po("R"+Et,2));(T.kjb(w.aAb(z).substr(0,6),"M  SAL")&&w.aAb(z).length>=17||T.kjb(w.aAb(z).substr(0,6),"M  SDT")&&w.aAb(z).length>=12||T.kjb(w.aAb(z).substr(0,6),"M  SED")&&w.aAb(z).length>=12)&&(!c&&(c=new y.fwb()),qt=(w.Wzb(7,10,w.aAb(z).length),w.aAb(z).substr(7,3)),h=(I=qt,y.fnb(y.Uvb(c,I))),h||(h=new e.Es(),y.Zvb(c,qt,h)),T.kjb(w.aAb(z).substr(0,6),"M  SAL")&&T.kjb(w.aAb(z).substr(10,w.aAb("  1").length),"  1")?h.a=T.Whb(T.Cjb((w.Wzb(13,17,w.aAb(z).length),w.aAb(z).substr(13,4))),P1,St):T.kjb(w.aAb(z).substr(0,6),"M  SDT")?h.b=T.zjb((w.Xzb(11,w.aAb(z).length+1),w.aAb(z).substr(11)),iP):T.kjb(w.aAb(z).substr(0,6),"M  SED")&&(h.c=T.Cjb((w.Xzb(11,w.aAb(z).length+1),w.aAb(z).substr(11))))),z=Ct.ngb(A);}}catch(L1){if(L1=M1(L1),Kt(L1,19))return g=L1,T.MB(g,(T.Zjb(),T.Yjb),""),false;throw Ut(L1);}if(c)for(b=(yt=new y.Oob(c).a.bd().Tb(),new y.Vob(yt));b.a.dd();)h=(B=b.a.ed(),B.hd()),h.b&&h.a!=0&&h.c!=null&&e.mn(t.j,h.a-1,h.c);return t.e&&(e.vs(t),e.es(t)),(t.i&1)!=0&&(t.f=e.lp(t.j)),e.js(t,u1),e.bw(t.j,15),true;},e.Bs=function(t,A){var n,P,r,i,D;for(t.b&&y.Svb(t.b),t.c&&y.Svb(t.c),r=0,P=Ct.ngb(A);P!=null&&T.kjb(w.aAb(P).substr(0,7),Ln);){for(P=T.Cjb((w.Xzb(7,w.aAb(P).length+1),w.aAb(P).substr(7)));D=w.aAb("-").length,T.kjb(w.aAb(P).substr(w.aAb(P).length-D,D),"-");){if(n=Ct.ngb(A),!T.kjb(w.aAb(n).substr(0,7),Ln))return  false;P=T.Cjb(T.gjb(T.Bjb(P,0,w.aAb(P).length-1),(w.Xzb(7,w.aAb(n).length+1),w.aAb(n).substr(7))));}if(T.kjb(w.aAb(P).substr(0,5),"BEGIN")){if(i=T.Cjb((w.Xzb(6,w.aAb(P).length+1),w.aAb(P).substr(6))),T.kjb(w.aAb(i).substr(0,4),"CTAB"))r=1;else if(T.kjb(w.aAb(i).substr(0,4),"ATOM"))r=2;else if(T.kjb(w.aAb(i).substr(0,4),"BOND"))r=3;else if(T.kjb(w.aAb(i).substr(0,6),"SGROUP"))r=4;else if(T.kjb(w.aAb(i).substr(0,10),"COLLECTION"))r=5;else return  false;}else if(T.kjb(w.aAb(P).substr(0,3),"END"))r=0;else if(r==1)e.ss(t,P);else if(r==2)e.os(t,P);else if(r==3)e.qs(t,P);else if(r==4)e.us(t,P);else if(r==5)e.rs(t,P);else return  false;P=Ct.ngb(A);}for(;P!=null&&!(T.kjb(w.aAb(P).substr(0,6),FA)||T.kjb(P,Qn));)P=Ct.ngb(A);return  true;},e.Cs=function(){this.i=0;},e.Ds=function(){this.i=1;},N(127,1,{},e.Cs,e.Ds),U.a=false,U.d=false,U.e=false,U.g=false,U.i=0,U.k=false,l.U7=O(127),e.Es=function(){},N(224,1,{224:1},e.Es),U.a=0,U.b=false,l.T7=O(224),e.Fs=function(t){},e.Gs=function(t,A){var n,P;for(P=l0.Nmb(t.a,A),n=w.aAb(P).length;n<10;n++)T.Ojb(t.b,32);T.Sjb(t.b,P);},e.Hs=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt;for(tt=(T.Zjb(),k1),T.Sjb(t.b,"M  V30 BEGIN CTAB"+tt),T.Sjb(t.b,K8+A.q+" "+A.r+" 0 0 0"+tt),T.Sjb(t.b,"M  V30 BEGIN ATOM"+tt),o=0;o<A.q;o++){if(T.Sjb(t.b,Ln+(o+1)),(A.v==null?null:A.v[o])!=null){for(h=A.v==null?null:A.v[o],ot=pt(J(A.B[o],1),0),T.Sjb(t.b,ot?" NOT[":" ["),V=0;V<h.length;V++)switch(V>0&&T.Sjb(t.b,","),I=(e.Rk(),e.Lk)[h[V]],w.aAb(I).length){case 2:case 3:case 1:T.Sjb(t.b,I);break;default:T.Sjb(t.b,"?");}T.Sjb(t.b,"]");}else pt(J(A.B[o],1),0)?T.Sjb(t.b," A"):A.C[o]>=129&&A.C[o]<=144||A.C[o]==154?T.Sjb(t.b," R#"):T.Sjb(t.b," "+(e.Rk(),e.Lk)[A.C[o]]);if(n?(T.Sjb(t.b," "+Rt(K0*t.c*A.J[o].a)/K0),T.Sjb(t.b," "+Rt(K0*t.c*-A.J[o].b)/K0),T.Sjb(t.b," "+Rt(K0*t.c*-A.J[o].c)/K0)):T.Sjb(t.b," 0 0 0"),T.Sjb(t.b," "+d.Math.abs(A.w[o])),A.s[o]!=0&&T.Sjb(t.b," CHG="+A.s[o]),(A.u[o]&48)!=0)switch(T.Sjb(t.b," RAD="),A.u[o]&48){case 16:T.Sjb(t.b,"1");break;case 32:T.Sjb(t.b,"2");break;case 48:T.Sjb(t.b,"3");}((A.u[o]&3)==1||(A.u[o]&3)==2)&&(T.Sjb(t.b," CFG="),(A.u[o]&3)==1?T.Sjb(t.b,"1"):T.Sjb(t.b,"2")),A.A[o]!=0&&T.Sjb(t.b," MASS="+A.A[o]),xt=((A.u[o]&cA)>>>27)-1,xt!=-1&&T.Sjb(t.b," VAL="+(xt==0?"-1":T.Aib(xt))),b=A.C[o],(b>=129&&b<=144||b==154)&&T.Sjb(t.b," RGROUPS=(1 "+(b==154?0:b>=142?b-141:b-125)+")"),S=J(X0,A.B[o]),et(S,384)==0?T.Sjb(t.b," HCOUNT=2"):et(S,128)==0?T.Sjb(t.b," HCOUNT=1"):et(S,1792)==0?T.Sjb(t.b," HCOUNT=-1"):et(S,1664)==0&&T.Sjb(t.b," HCOUNT=1"),yt=J(A.B[o],6144),et(yt,0)!=0&&(pt(J(yt,Z1),0)?T.Sjb(t.b," SUBST="+(A.e[o]+1)):T.Sjb(t.b," SUBST=-1")),Dt=J(A.B[o],120),et(Dt,0)!=0&&(et(Dt,112)==0?T.Sjb(t.b," RBCNT=-1"):et(Dt,8)==0||et(Dt,104)==0?T.Sjb(t.b," RBCNT=2"):et(Dt,88)==0?T.Sjb(t.b," RBCNT=3"):et(Dt,56)==0&&T.Sjb(t.b," RBCNT=4")),T.Sjb(t.b,tt);}for(T.Sjb(t.b,"M  V30 END ATOM"+tt),T.Sjb(t.b,"M  V30 BEGIN BOND"+tt),g=0;g<A.r;g++){switch(T.Sjb(t.b,Ln+(g+1)),A.H[g]){case 1:Qt=1,ct=0;break;case 2:Qt=2,ct=0;break;case 4:Qt=3,ct=0;break;case 129:Qt=1,ct=3;break;case 257:Qt=1,ct=1;break;case 386:Qt=2,ct=2;break;case 8:Qt=4,ct=0;break;case 16:Qt=9,ct=0;break;default:Qt=1,ct=0;}k=A.G[g]&31,k!=0&&(k==8?Qt=4:k==3?Qt=5:k==9?Qt=6:k==10?Qt=7:Qt=8),T.Sjb(t.b," "+Qt+" "+(A.D[0][g]+1)+" "+(A.D[1][g]+1)),ct!=0&&T.Sjb(t.b," CFG="+ct),lt=A.G[g]&384,wt=lt==0?0:lt==256?1:2,wt!=0&&T.Sjb(t.b," TOPO="+wt),T.Sjb(t.b,tt);}for(T.Sjb(t.b,"M  V30 END BOND"+tt),p=false,Q=0;Q<A.q;Q++)if((A.t==null||A.t[Q]==null?null:T.ijb((Z=A.t[Q],S1.Mmb(),Z)))!=null){p=true;break;}if(p){for(T.Sjb(t.b,"M  V30 BEGIN SGROUP"+tt),M=0,L=0;L<A.q;L++)(A.t==null||A.t[L]==null?null:T.ijb((z=A.t[L],S1.Mmb(),z)))!=null&&(++M,T.Sjb(t.b,Ln+M+" DAT "+M+" ATOMS=(1 "+(L+1)+') FIELDNAME="'+iP+'" -'+tt),T.Sjb(t.b,'M  V30 FIELDDISP="'),e.Gs(t,n?t.c*A.J[L].a:0),e.Gs(t,n?t.c*A.J[L].b:0),T.Sjb(t.b,'    DA    ALL  1       5" FIELDDATA="'+(A.t==null||A.t[L]==null?null:T.ijb((Z=A.t[L],S1.Mmb(),Z)))+'"'+tt));T.Sjb(t.b,"M  V30 END SGROUP"+tt);}for(ut=false,P=0,G=j(l.j7,H,6,32,15,1),i=j(l.j7,H,6,32,15,1),s=0;s<A.f;s++)((A.u[s]&3)==1||(A.u[s]&3)==2)&&(ut=true,bt=(A.u[s]&Q1)>>18,bt==1?++i[(A.u[s]&Q1)>>18!=1&&(A.u[s]&Q1)>>18!=2?-1:(A.u[s]&F0)>>20]:bt==2?++G[(A.u[s]&Q1)>>18!=1&&(A.u[s]&Q1)>>18!=2?-1:(A.u[s]&F0)>>20]:++P);for(r=0,it=j(l.j7,H,6,32,15,1),D=j(l.j7,H,6,32,15,1),B=0;B<A.g;B++)e.fm(A,B)!=2&&((A.F[B]&3)==1||(A.F[B]&3)==2)&&(ut=true,bt=(A.F[B]&768)>>8,bt==1?++D[(A.F[B]&768)>>8!=1&&(A.F[B]&768)>>8!=2?-1:(A.F[B]&E0)>>10]:bt==2?++it[(A.F[B]&768)>>8!=1&&(A.F[B]&768)>>8!=2?-1:(A.F[B]&E0)>>10]:++r);if(ut){if(T.Sjb(t.b,"M  V30 BEGIN COLLECTION"+tt),P!=0){for(T.Sjb(t.b,"M  V30 MDLV30/STEABS ATOMS=("+P),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&Q1)>>18==0&&T.Sjb(t.b," "+(u+1));T.Sjb(t.b,")"+tt);}if(r!=0){for(T.Sjb(t.b,"M  V30 MDLV30/STEABS BONDS=("+r),c=0;c<A.g;c++)e.fm(A,c)!=2&&((A.F[c]&3)==1||(A.F[c]&3)==2)&&(A.F[c]&768)>>8==0&&T.Sjb(t.b," "+(c+1));T.Sjb(t.b,")"+tt);}for(q=0;q<32;q++){if(G[q]!=0){for(T.Sjb(t.b,Y8+(q+1)+" ATOMS=("+G[q]),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&Q1)>>18==2&&((A.u[u]&Q1)>>18!=1&&(A.u[u]&Q1)>>18!=2?-1:(A.u[u]&F0)>>20)==q&&T.Sjb(t.b," "+(u+1));T.Sjb(t.b,")"+tt);}if(i[q]!=0){for(T.Sjb(t.b,X8+(q+1)+" ATOMS=("+i[q]),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&Q1)>>18==1&&((A.u[u]&Q1)>>18!=1&&(A.u[u]&Q1)>>18!=2?-1:(A.u[u]&F0)>>20)==q&&T.Sjb(t.b," "+(u+1));T.Sjb(t.b,")"+tt);}if(it[q]!=0){for(T.Sjb(t.b,Y8+(q+1)+" BONDS=("+it[q]),c=0;c<A.g;c++)e.fm(A,c)!=2&&((A.F[c]&3)==1||(A.F[c]&3)==2)&&(A.F[c]&768)>>8==2&&((A.F[c]&768)>>8!=1&&(A.F[c]&768)>>8!=2?-1:(A.F[c]&E0)>>10)==q&&T.Sjb(t.b," "+(c+1));T.Sjb(t.b,")"+tt);}if(D[q]!=0){for(T.Sjb(t.b,X8+(q+1)+" BONDS=("+D[q]),c=0;c<A.g;c++)e.fm(A,c)!=2&&((A.F[c]&3)==1||(A.F[c]&3)==2)&&(A.F[c]&768)>>8==1&&((A.F[c]&768)>>8!=1&&(A.F[c]&768)>>8!=2?-1:(A.F[c]&E0)>>10)==q&&T.Sjb(t.b," "+(c+1));T.Sjb(t.b,")"+tt);}}T.Sjb(t.b,"M  V30 END COLLECTION"+tt);}T.Sjb(t.b,Lr+tt);},e.Is=function(){e.Fs(this),this.b=new T.Vjb();},e.Js=function(t){e.Ks.call(this,t);},e.Ks=function(t){e.Ls.call(this,t,new T.Vjb());},e.Ls=function(t,A){var n,P,r,i;e.Fs(this),this.a=new l0.Smb((P=rP,P)),e.bw(t,15),i=(T.Zjb(),k1),this.b=A,r=t.P!=null?t.P:"",T.Sjb(this.b,r+i),T.Sjb(this.b,"Actelion Java MolfileCreator 2.0"+i+i),T.Sjb(this.b,"  0  0  0  0  0  0              0 V3000"+i),this.c=1,n=e.Ns(t),n&&(this.c=e.Ms(t)),e.Hs(this,t,n),T.Sjb(this.b,FA+i);},e.Ms=function(t){var A,n,P,r,i,D,u,o,Q;if(Q=1,P=e.Wl(t,t.q,t.r,(e.Rk(),e.Qk)),P!=0)(P<1||P>3)&&(Q=1.5/P);else {for(o=_8,A=1;A<t.q;A++)for(n=0;n<A;n++)i=t.J[n].a-t.J[A].a,D=t.J[n].b-t.J[A].b,u=t.J[n].c-t.J[A].c,r=i*i+D*D+u*u,o>r&&(o=r);Q=3/d.Math.max(.75,o);}return Q;},e.Ns=function(t){var A;if(t.q==1)return  true;for(A=1;A<t.q;A++)if(t.J[A].a!=t.J[0].a||t.J[A].b!=t.J[0].b||t.J[A].c!=t.J[0].c)return  true;return  false;},e.Os=function(t,A){var n;return n=new e.Is(),n.c=A,e.bw(t,15),e.Hs(n,t,true),n.b.a;},N(150,1,{},e.Is,e.Js),U.c=1,l.V7=O(150),e.Rs=function(){e.Rs=D1,e.Ps=v(F(l.Ybb,2),c0,13,0,[v(F(l.Ybb,1),st,2,6,["Acyl","gCaHA`AIf`@"]),v(F(l.Ybb,1),st,2,6,["Alloc","gNph@l@ILzuR@@"]),v(F(l.Ybb,1),st,2,6,["Allyl","Al"]),v(F(l.Ybb,1),st,2,6,["Benzyl;Bn","daD@`F@DjUZxHH@@"]),v(F(l.Ybb,1),st,2,6,["Boc","daxD`@S@AIgijj@@"]),v(F(l.Ybb,1),st,2,6,["BOM;BzOM","deTH`@H@Re[TYj`@@@"]),v(F(l.Ybb,1),st,2,6,["Bs","dmtDpAdLS`aPAIe]jf`@e`@@"]),v(F(l.Ybb,1),st,2,6,["Bt","dew@`@aJ@DiY]paej`@@@"]),v(F(l.Ybb,1),st,2,6,["Btm","did@P@BJ@Die_ahH@@@"]),v(F(l.Ybb,1),st,2,6,["Butyl;nButyl","gJPHB@IRuP@"]),v(F(l.Ybb,1),st,2,6,["Benzoyl;Bz","didH`@p@RYm^Eh@@@@"]),v(F(l.Ybb,1),st,2,6,["Bzh","dg|@`N@LdbRbtJUB]aAP@@@@"]),v(F(l.Ybb,1),st,2,6,["Cbz","dmtD`@S@AIgYVUZh@@@@"]),v(F(l.Ybb,1),st,2,6,["cButyl","gKPHL@IThuT@@"]),v(F(l.Ybb,1),st,2,6,["cHeptyl","daD@`L@DjWVzjj`@"]),v(F(l.Ybb,1),st,2,6,["cHexyl","gOpHL@IToWUU@@"]),v(F(l.Ybb,1),st,2,6,["cOctyl","did@`L@DjWWajjj@@"]),v(F(l.Ybb,1),st,2,6,["cPentyl","gFpHL@ITimUP@"]),v(F(l.Ybb,1),st,2,6,["cPropyl","gBPHL@Qxjh@"]),v(F(l.Ybb,1),st,2,6,["COOH","gC`h@l@ILt@@ !Bb@K~@Hc}"]),v(F(l.Ybb,1),st,2,6,["DEAE","daz@`@x@RiUjj`@"]),v(F(l.Ybb,1),st,2,6,["DEIPS","diD@P@\\B@DjfVjj`@"]),v(F(l.Ybb,1),st,2,6,["DMIPS","gNpD@xD@RjZjh@"]),v(F(l.Ybb,1),st,2,6,["DMPM","dcLD`@kPCIEMDdcttDDT@@"]),v(F(l.Ybb,1),st,2,6,["DMPS","deT@P@\\B@LddTjPsU@@@@"]),v(F(l.Ybb,1),st,2,6,["DMTr","fak@b@@Mt@ISZ{SMjo{NQKfm@AU@@@E@@@@"]),v(F(l.Ybb,1),st,2,6,["DNP","dkmB`hdDt~@HeNfS{HihheCAUhBHX@@"]),v(F(l.Ybb,1),st,2,6,["DNS;Dan","fhi`a@KPP@HH@YIHYheEhYKQgKP@@QP@@@"]),v(F(l.Ybb,1),st,2,6,["DPIPS","fdyAA@H@\\B@FRRIQSQIHzp_Qjh@h@@@@@"]),v(F(l.Ybb,1),st,2,6,["DPTBS","fleAA@H@\\B@FRRIQSRIIWNbEMU@EP@@@@@"]),v(F(l.Ybb,1),st,2,6,["DTBMS","dmT@P@\\B@Djffjjjh@@"]),v(F(l.Ybb,1),st,2,6,["Ethyl","eMBD@ch@"]),v(F(l.Ybb,1),st,2,6,["Fmoc","fde@b@@Hp@IL{LrjxeVCzKUT@@@P@@@"]),v(F(l.Ybb,1),st,2,6,["iAmyl","gGPHJ@YIDZj@@"]),v(F(l.Ybb,1),st,2,6,["iButyl","gJPHB@ITuP@"]),v(F(l.Ybb,1),st,2,6,["Im","gFtHAj@IRnKSP@"]),v(F(l.Ybb,1),st,2,6,["iPropyl","gC`HL@Qz`@"]),v(F(l.Ybb,1),st,2,6,["MDIPS","diD@P@\\B@DjfZjj`@"]),v(F(l.Ybb,1),st,2,6,["MDPS","foA@A@@NA@CIIEEBdeeVLzj@@@@@@"]),v(F(l.Ybb,1),st,2,6,["Methyl","eFBH@c@@"]),v(F(l.Ybb,1),st,2,6,["MEM","gNphAR@IRoUT@@"]),v(F(l.Ybb,1),st,2,6,["Mes","deT@`J@DjY{[`bB`@@"]),v(F(l.Ybb,1),st,2,6,["MMTr","ffcAB@@Z@Dim]ifuWYrI\\uh@Jh@@@@@@"]),v(F(l.Ybb,1),st,2,6,["MOM","gCaHA`AJZ`@"]),v(F(l.Ybb,1),st,2,6,["MPM;PMB","deTH`@d@Rfuunh@J@@"]),v(F(l.Ybb,1),st,2,6,["Ms","gJPdH`DD@cuh@"]),v(F(l.Ybb,1),st,2,6,["MTM","gC`D@DX@Rfh@"]),v(F(l.Ybb,1),st,2,6,["mTolyl","daD@`N@DjWjXHB@@"]),v(F(l.Ybb,1),st,2,6,["N3","gClHaE`@RnReX@"]),v(F(l.Ybb,1),st,2,6,["nAmyl;Amyl","gGPHJ@IRmU@@"]),v(F(l.Ybb,1),st,2,6,["neoAm","gGPHJ@IUMU@@"]),v(F(l.Ybb,1),st,2,6,["NO2,Nitro","gChhhE`BRnRYh@"]),v(F(l.Ybb,1),st,2,6,["Np","deVDaHAI@HeNR[e_aZ@B@@"]),v(F(l.Ybb,1),st,2,6,["nPropyl;Propyl","gC`HL@IST@@"]),v(F(l.Ybb,1),st,2,6,["oTolyl","daD@`J@DjYvxH`@@"]),v(F(l.Ybb,1),st,2,6,["Phenyl","gOpHL@IToVD@@@"]),v(F(l.Ybb,1),st,2,6,["Pht","dcLL`@RU@Dfyed]ZBA`@@"]),v(F(l.Ybb,1),st,2,6,["Piv;Pv","gNqHA`AIffj`@"]),v(F(l.Ybb,1),st,2,6,["PMBM","dcLD`@T`AJUm]FZh@J@@"]),v(F(l.Ybb,1),st,2,6,["PNB","dcNLaHAEt@bTyInUvxV`@f@@"]),v(F(l.Ybb,1),st,2,6,["Poc","didD`@S@AIgexVjj`@"]),v(F(l.Ybb,1),st,2,6,["PPi","diDFsHSB[`|J|A@Lxn{lddqdZih@@"]),v(F(l.Ybb,1),st,2,6,["pTolyl","daD@`N@DjWzXHB@@"]),v(F(l.Ybb,1),st,2,6,["sAmyl","gGPHL@YIDZj@@"]),v(F(l.Ybb,1),st,2,6,["sButyl","gJPHL@ITuP@"]),v(F(l.Ybb,1),st,2,6,["SEM","diDHPFApD@rRQUJjj`@"]),v(F(l.Ybb,1),st,2,6,["SES","dedDpHP@``AgCIICeHmUT@@"]),v(F(l.Ybb,1),st,2,6,["SO3H","gJQdHl@``D^m@@"]),v(F(l.Ybb,1),st,2,6,["tAmyl","gGPHB@IUMU@@"]),v(F(l.Ybb,1),st,2,6,["TBDMS;TBS","dax@P@\\B@Djfjjh@@"]),v(F(l.Ybb,1),st,2,6,["TBDPS","fdy@A@@NA@CIIEEEIde]XOhuPAT@@@@@"]),v(F(l.Ybb,1),st,2,6,["TBMPS","dg\\HPHApH@rRQJJPjg]UAT@@@"]),v(F(l.Ybb,1),st,2,6,["tButyl,tBu","gJPHB@Q}T@@"]),v(F(l.Ybb,1),st,2,6,["TDS","ded@P@\\B@LddTeeUUP@@"]),v(F(l.Ybb,1),st,2,6,["Tf","daxDhHP@``BiAiCiCIICHmU@@"]),v(F(l.Ybb,1),st,2,6,["TFA","gNqBJIARFdF@YEHYUL@@"]),v(F(l.Ybb,1),st,2,6,["Thexyl","gNpHB@IUMUT@@"]),v(F(l.Ybb,1),st,2,6,["THF","gFqH@PAJYujj@@"]),v(F(l.Ybb,1),st,2,6,["THP","gOqH@PAJYZzjh@"]),v(F(l.Ybb,1),st,2,6,["TIPS","dmT@P@\\B@DjfYjjjh@@"]),v(F(l.Ybb,1),st,2,6,["TMS","gJPD@xD@czh@"]),v(F(l.Ybb,1),st,2,6,["Tosyl;Ts","dmtDPHP@``CIICLeaeZ@B@@"]),v(F(l.Ybb,1),st,2,6,["Troc","diDDHJxHaHcH`PCHiBeJjf@@"]),v(F(l.Ybb,1),st,2,6,["Trt","fbm@B@A@FRQIRKQPiIZdoIcdHJ`@@@@@@"]),v(F(l.Ybb,1),st,2,6,["Xyl","did@`J@DjYynBHH@@"])]);},e.Ss=function(){e.Rs();var t,A,n,P,r,i,D,u,o;for(e.Qs=new y.fwb(),D=e.Ps,u=0,o=D.length;u<o;++u)for(i=D[u],r=T.yjb(i[0],";"),A=r,n=0,P=A.length;n<P;++n)t=A[n],y.Zvb(e.Qs,T.wjb(w.aAb(t).toLowerCase(),"-"),i[1]);},e.Ts=function(t){e.Rs();var A,n,P,r,i,D,u,o,Q,L;if(t==null||w.aAb(t).length==0)return null;for(D=null,t=T.wjb(w.aAb(t).toLowerCase(),"-"),o=e.Ps,Q=0,L=o.length;Q<L;++Q)for(u=o[Q],i=T.yjb(u[0],";"),n=i,P=0,r=n.length;P<r;++P){if(A=n[P],T.kjb(T.wjb(w.aAb(A).toLowerCase(),"-"),t))return A;if(T.zjb(T.wjb(w.aAb(A).toLowerCase(),"-"),t))if(D==null)D=A;else return "";}return D;},e.Qs=null,e.Ws=function(){var t,A,n,P;for(this.b=new y.ztb(),this.c=new y.ztb(),this.d=new y.ztb(),e.Us=v(F(l.H7,1),C1,18,0,[new e.mk(1,"Hydrogen","H",2.2),new e.mk(2,"Helium","He",0),new e.mk(3,"Lithium","Li",.98),new e.mk(4,"Beryllium","Be",1.57),new e.mk(5,"Boron","B",2.04),new e.mk(6,"Carbon","C",2.55),new e.mk(7,"Nitrogen","N",3.04),new e.mk(8,"Oxygen","O",3.44),new e.mk(9,"Fluorine","F",3.98),new e.mk(10,"Neon","Ne",0),new e.mk(11,"Sodium","Na",.93),new e.mk(12,"Magnesium","Mg",1.31),new e.mk(13,"Aluminum","Al",1.61),new e.mk(14,"Silicon","Si",1.9),new e.mk(15,"Phosphorus","P",2.19),new e.mk(16,"Sulfur","S",2.58),new e.mk(17,"Chlorine","Cl",3.16),new e.mk(18,"Argon","Ar",0),new e.mk(19,"Potassium","K",1),new e.mk(20,"Calcium","Ca",1.36),new e.mk(21,"Scandium","Sc",1.54),new e.mk(22,"Titanium","Ti",1.63),new e.mk(23,"Vanadium","V",1.66),new e.mk(24,"Chromium","Cr",1.55),new e.mk(25,"Manganese","Mn",1.83),new e.mk(26,"Iron","Fe",1.88),new e.mk(27,"Cobalt","Co",1.91),new e.mk(28,"Nickel","Ni",1.9),new e.mk(29,"Copper","Cu",1.65),new e.mk(30,"Zinc","Zn",1.81),new e.mk(31,"Gallium","Ga",2.01),new e.mk(32,"Germanium","Ge",0),new e.mk(33,"Arsenic","As",2.18),new e.mk(34,"Selenium","Se",2.55),new e.mk(35,"Bromine","Br",0),new e.mk(36,"Krypton","Kr",2.96),new e.mk(37,"Rubidium","Rb",0),new e.mk(38,"Strontium","Sr",.82),new e.mk(39,"Yttrium","Y",.95),new e.mk(40,"Zirconium","Zr",1.22),new e.mk(41,"Niobium","Nb",1.33),new e.mk(42,"Molybdenum","Mo",1.6),new e.mk(43,"Technetium","Tc",2.16),new e.mk(44,"Ruthenium","Ru",1.9),new e.mk(45,"Rhodium","Rh",2.2),new e.mk(46,"Palladium","Pd",2.28),new e.mk(47,"Silver","Ag",2.2),new e.mk(48,"Cadmium","Cd",1.93),new e.mk(49,"Indium","In",1.69),new e.mk(50,"Tin","Sn",1.78),new e.mk(51,"Antimony","Sb",1.96),new e.mk(52,"Tellurium","Te",0),new e.mk(53,"Iodine","I",2.05),new e.mk(54,"Xenon","Xe",2.1),new e.mk(55,"Cesium","Cs",2.66),new e.mk(56,"Barium","Ba",2.6),new e.mk(57,"Lanthanum","La",.79),new e.mk(58,"Cerium","Ce",.89),new e.mk(59,"Praesodymium","Pr",1.1),new e.mk(60,"Neodymium","Nd",1.12),new e.mk(61,"Promethium","Pm",1.13),new e.mk(62,"Samarium","Sm",1.14),new e.mk(63,"Europium","Eu",0),new e.mk(64,"Gadolinium","Gd",1.17),new e.mk(65,"Terbium","Tb",0),new e.mk(66,"Dyprosium","Dy",1.2),new e.mk(67,"Holmium","Ho",0),new e.mk(68,"Erbium","Er",1.22),new e.mk(69,"Thulium","Tm",1.23),new e.mk(70,"Ytterbium","Yb",1.24),new e.mk(71,"Lutetium","Lu",1.25),new e.mk(72,"Hafnium","Hf",0),new e.mk(73,"Tantalium","Ta",1.27),new e.mk(74,"Wolfram","W",1.3),new e.mk(75,"Rhenium","Re",1.5),new e.mk(76,"Osmium","Os",2.36),new e.mk(77,"Iridium","Ir",1.9),new e.mk(78,"Platinum","Pt",2.2),new e.mk(79,"Gold","Au",2.2),new e.mk(80,"Mercury","Hg",2.28),new e.mk(81,"Thallium","Tl",2.54),new e.mk(82,"Lead","Pb",2),new e.mk(83,"Bismuth","Bi",1.62),new e.mk(84,"Polonium","Po",2.33),new e.mk(85,"Astatine","At",2.02),new e.mk(86,"Radon","Rn",2),new e.mk(87,"Francium","Fr",2.2),new e.mk(88,"Radium","Ra",0),new e.mk(89,"Actinium","Ac",.7),new e.mk(90,"Thorium","Th",.89),new e.mk(91,"Protactinium","Pa",1.1),new e.mk(92,"Uranium","U",1.3),new e.mk(93,"Neptunium","Np",1.5),new e.mk(94,"Plutonium","Pu",1.38),new e.mk(95,"Americium","Am",1.36),new e.mk(96,"Curium","Cm",1.28),new e.mk(97,"Berkelium","Bk",1.3),new e.mk(98,"Californium","Cf",1.3),new e.mk(99,"Einsteinium","E",1.3),new e.mk(100,"Fermium","Fm",1.3),new e.mk(101,"Mendelevium","Mv",1.3)]),A=e.Us,n=0,P=A.length;n<P;++n)t=A[n],y.Anb(this.b,T.Aib(t.c),t),y.Bnb(this.c,t.b,t),y.Bnb(this.d,w.aAb(t.d).toLowerCase(),t);this.a=j(l.Feb,ht,6,e.Us.length,16,1),this.a[3]=true,this.a[11]=true,this.a[19]=true,this.a[37]=true,this.a[55]=true,this.a[87]=true;},e.Xs=function(t){return y.xnb((!e.Vs&&(e.Vs=new e.Ws()),e.Vs).b,T.Aib(t));},N(393,1,{},e.Ws),U.pb=function(){return "";},l.W7=O(393),e.Zs=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g;for(Q=t.i.M,L=0,i=0;i<n;i++)Q>A[i]&&(Q=A[i],L=i);for(g=j(l.j7,H,6,n,15,1),o=L>0?L-1:n-1,s=L<n-1?L+1:0,D=A[o]<A[s],r=0;r<n;r++)g[r]=A[L],D?--L<0&&(L=n-1):++L==n&&(L=0);for(c=new y.Bqb(t.j);c.a<c.c.a.length;)if(b=y.Aqb(c),b.length==n){for(P=true,u=0;u<n;u++)if(b[u]!=g[u]){P=false;break;}if(P)return;}y.sj(t.j,g),h=e.jt(t,g),y.sj(t.k,h),e.Bt(t,g,h);},e.$s=function(t,A,n){var P,r,i,D,u,o,Q;for(o=j(l.j7,H,6,t.g,15,1),D=j(l.j7,H,6,t.g,15,1),Q=j(l.Feb,ht,6,t.i.f,16,1),P=e.Zl(t.i,0,A),r=e.Zl(t.i,1,A),o[0]=P,o[1]=r,D[1]=-1,Q[r]=true,u=1;u>=1;){if(++D[u],D[u]==e.$o(t.i,o[u])){Q[o[u]]=false,--u;continue;}if(i=e.Zo(t.i,o[u],D[u]),!(Q[i]||n[i])){if(i==P&&u>1){if(e.Zs(t,o,u+1),t.j.a.length>=i0)return;continue;}u+1<t.g&&(++u,o[u]=i,Q[i]=true,D[u]=-1);}}},e._s=function(t,A,n,P,r,i,D,u){var o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt;for(Qt=y.xj(t.j,A),s=Qt,h=0,b=s.length;h<b;++h)if(L=s[h],!e.xt(t,L))return  true;for(ut=y.xj(t.k,A),Dt=ut.length,g=0,Q=0,bt=false,I=0;I<Dt;I++)g<<=1,Q<<=1,e.zt(t,ut[I])?g|=1:u&&e.wt(t,ut[I])?(g|=1,Q|=1):(o=n[A][I],o!=-1&&(P[o]?r[o]&&(g|=1,i[o]||(Q|=1)):bt=true));switch(S=false,Dt){case 3:for(B=v(F(l.j7,1),H,6,15,[2,1,4]),S=true,q=0;q<3;q++)(g&B[q])==B[q]&&(e.Tl(t.i,Qt[q])==6&&e.Bl(t.i,Qt[q])==1||e.Tl(t.i,Qt[q])==5&&e.Bl(t.i,Qt[q])==0)&&(r[A]=true,D[A]=q,(Q&B[q])==0&&(S=false));break;case 5:for(k=v(F(l.j7,1),H,6,15,[10,5,18,9,20]),S=true,it=0;it<5;it++)if((g&k[it])==k[it])switch(e.Tl(t.i,Qt[it])){case 6:e.Bl(t.i,Qt[it])==-1&&(r[A]=true,D[A]=it,(Q&k[it])==0&&(S=false));break;case 7:e.Bl(t.i,Qt[it])<=0&&(r[A]=true,D[A]=it);break;case 8:r[A]=true,D[A]=it;break;case 16:case 34:case 52:e.$o(t.i,Qt[it])==2&&(r[A]=true,D[A]=it);}break;case 6:S=true,(g&21)==21&&(r[A]=true,(Q&21)==0&&(S=false)),(g&42)==42&&(r[A]=true,(Q&42)==0&&(S=false));break;case 7:for(M=v(F(l.j7,1),H,6,15,[42,21,74,37,82,41,84]),S=true,p=0;p<7;p++)if((g&M[p])==M[p]){if(e.Tl(t.i,Qt[p])==6&&(e.Bl(t.i,Qt[p])==1||u&&e.mt(t,Qt[p]))||e.Tl(t.i,Qt[p])==5&&e.Bl(t.i,Qt[p])==0)r[A]=true,D[A]=p,(Q&M[p])==0&&(S=false);else for(V=0;V<2;V++)if(ct=(p+6+V)%7,lt=ut[ct],yt=n[A][ct],yt!=-1&&!r[yt]&&y.xj(t.k,yt).length==5){for(wt=y.xj(t.k,yt),Z=0;wt[Z]!=lt;)++Z;for(z=false,G=0,tt=1;tt<=4;tt++){if(c=wt[(Z+tt)%5],ot=e.fm(t.i,c)>1||e.im(t.i,c)==8||u&&e.wt(t,c),ot&&z){G=0;break;}z=ot,z&&++G;}if(G==2){r[A]=true,r[yt]=true,i[A]=true,i[yt]=true,y.sj(t.b,T.Aib(lt));break;}}}}return r[A]&&!S&&(i[A]=true),r[A]?true:!bt;},e.at=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c,g,B;for(i=j(l.j7,E1,7,t.j.a.length,0,2),Q=0;Q<t.j.a.length;Q++)for(i[Q]=j(l.j7,H,6,y.xj(t.j,Q).length,15,1),L=0;L<y.xj(t.j,Q).length;L++)i[Q][L]=-1;for(g=j(l.j7,H,6,t.i.g,15,1),b=0;b<t.k.a.length;b++)if(c=y.xj(t.k,b),c.length==3||c.length>=5&&c.length<=7)for(o=0;o<c.length;o++)u=c[o],e.$o(t.i,e.Zl(t.i,0,u))==3&&e.$o(t.i,e.Zl(t.i,1,u))==3&&(g[u]>0?(i[g[u]>>>16][g[u]&32767]=b,i[b][o]=g[u]>>>16):g[u]=(b<<16)+LA+o);for(D=j(l.Feb,ht,6,t.j.a.length,16,1),B=0,s=-1;B>s;)for(s=B,h=0;h<t.j.a.length;h++)D[h]||e._s(t,h,i,D,A,n,P,r)&&(D[h]=true,++B);},e.bt=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;for(r=e.Zl(t.i,0,A),i=e.Zl(t.i,1,A),o=j(l.j7,H,6,t.i.f,15,1),Q=j(l.j7,H,6,t.i.f,15,1),L=j(l.j7,H,6,t.i.f,15,1),o[0]=r,o[1]=i,Q[r]=1,Q[i]=2,L[r]=-1,L[i]=r,u=1,s=1;u<=s;){for(h=0;h<e.$o(t.i,o[u]);h++){if(D=e.Zo(t.i,o[u],h),u>1&&D==r){for(c=j(l.j7,H,6,Q[o[u]],15,1),P=o[u],b=0;b<c.length;b++)c[b]=P,P=L[P];return c;}Q[D]==0&&!n[D]&&(o[++s]=D,Q[D]=Q[o[u]]+1,L[D]=o[u]);}++u;}return null;},e.ct=function(t,A,n){var P,r;for(r=y.xj(t.j,A),P=0;P<r.length;P++)if(n==r[P])return P;return  -1;},e.dt=function(t,A){return t.a[A]&n1;},e.et=function(t,A,n){var P,r;for(r=y.xj(t.k,A),P=0;P<r.length;P++)if(n==r[P])return P;return  -1;},e.ft=function(t,A){return t.c[A]&n1;},e.gt=function(t,A){return t.d[A];},e.ht=function(t,A){return y.xj(t.j,A);},e.it=function(t,A){return y.xj(t.k,A);},e.jt=function(t,A){var n,P,r,i,D;for(i=A.length,D=j(l.j7,H,6,i,15,1),P=0;P<i;P++)for(n=P==i-1?A[0]:A[P+1],r=0;r<e.$o(t.i,A[P]);r++)if(e.Zo(t.i,A[P],r)==n){D[P]=e._o(t.i,A[P],r);break;}return D;},e.kt=function(t,A){return y.xj(t.k,A).length;},e.lt=function(t,A,n){var P;for(P=0;P<t.k.a.length;P++)if(e.rt(t,P,A)&&e.rt(t,P,n))return P;return  -1;},e.mt=function(t,A){var n;for(n=0;n<e.$o(t.i,A);n++)if(e.ap(t.i,A,n)==2&&e.Tl(t.i,e.Zo(t.i,A,n))==8)return  true;return  false;},e.nt=function(t,A){return t.e[A];},e.ot=function(t,A){return (t.a[A]&q1)!=0;},e.pt=function(t,A){return (t.c[A]&q1)!=0;},e.qt=function(t,A,n){var P,r,i,D;for(r=y.xj(t.j,A),i=0,D=r.length;i<D;++i)if(P=r[i],n==P)return  true;return  false;},e.rt=function(t,A,n){var P,r,i,D;for(r=y.xj(t.k,A),i=0,D=r.length;i<D;++i)if(P=r[i],n==P)return  true;return  false;},e.st=function(t,A){return t.f[A];},e.tt=function(t,A){return (t.a[A]&n0)!=0;},e.ut=function(t,A){return (t.c[A]&n0)!=0;},e.vt=function(t,A){return (t.a[A]&sA)!=0;},e.wt=function(t,A){var n,P,r,i,D,u;for(D=0;D<2;D++)if(n=e.Zl(t.i,D,A),e.Tl(t.i,n)==7&&e.$o(t.i,n)==2&&(P=e.Zl(t.i,1-D,A),e.Tl(t.i,P)==6)){for(u=0;u<e.$o(t.i,P);u++)if(r=e.Zo(t.i,P,u),i=e._o(t.i,P,u),(e.Tl(t.i,r)==8||e.Tl(t.i,r)==16)&&e.fm(t.i,i)==2&&e.$o(t.i,r)==1)return  true;}return  false;},e.xt=function(t,A){var n,P,r,i,D;if(t.i.K){if(pt(J(e.Ol(t.i,A),1),0))return  true;if(D=e.Jl(t.i,A),D!=null){for(P=D,r=0,i=P.length;r<i;++r)if(n=P[r],n==5||n==6||n==7||n==8||n==15||n==16||n==33||n==34)return  true;return  false;}}return e.Ft(e.Tl(t.i,A));},e.yt=function(t,A){var n,P,r,i,D;if(t.i.K){if(pt(J(e.Ol(t.i,A),1),0))return  false;if(n=e.Jl(t.i,A),n!=null){for(r=n,i=0,D=r.length;i<D;++i)if(P=r[i],!e.qo(P))return  false;return  true;}}return e.qo(e.Tl(t.i,A));},e.zt=function(t,A){return e.fm(t.i,A)>1||e.im(t.i,A)==8;},e.At=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;for(k=0;k<t.e.length;k++)if(t.e[k]){for(B=false,P=y.xj(t.j,k),i=0,u=P.length;i<u;++i)A=P[i],t.a[A]|=q1,e.yt(t,A)&&(B=true);for(L=y.xj(t.k,k),h=0,g=L.length;h<g;++h)o=L[h],t.c[o]|=q1;if(t.f[k]){for(n=y.xj(t.j,k),r=0,D=n.length;r<D;++r)A=n[r],t.a[A]|=n0;for(Q=y.xj(t.k,k),s=0,c=Q.length;s<c;++s)o=Q[s],t.c[o]|=n0;}if(B){for(n=y.xj(t.j,k),r=0,D=n.length;r<D;++r)A=n[r],t.a[A]|=sA;for(Q=y.xj(t.k,k),s=0,c=Q.length;s<c;++s)o=Q[s],t.c[o]|=sA;}}for(b=new y.Bqb(t.b);b.a<b.c.a.length;)o=y.Aqb(b).a,t.c[o]=0;},e.Bt=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h;for(h=A.length,r=A,i=0,D=r.length;i<D;++i)P=r[i],s=t.a[P]&n1,(s==0||s>h)&&(t.a[P]&=fA,t.a[P]|=h);for(o=n,Q=0,L=o.length;Q<L;++Q)u=o[Q],s=t.c[u]&n1,(s==0||s>h)&&(t.c[u]&=fA,t.c[u]|=h);},e.Ct=function(t,A,n){var P;for(P=y.xj(t.k,A).length;n>=P;)n-=P;for(;n<0;)n+=P;return n;},e.Dt=function(t,A){e.Et.call(this,t,A);},e.Et=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;this.i=t,this.g=7,this.j=new y.Mj(),this.k=new y.Mj(),this.b=new y.Mj(),this.a=j(l.j7,H,6,this.i.f,15,1),this.c=j(l.j7,H,6,this.i.g,15,1),this.i.$b(1),h=j(l.Feb,ht,6,this.i.f,16,1),b=j(l.Feb,ht,6,this.i.g,16,1);do for(D=false,n=0;n<this.i.f;n++)if(!h[n]){for(B=0,s=0;s<e.$o(this.i,n);s++)h[e.Zo(this.i,n,s)]||++B;if(B<2){for(h[n]=true,L=0;L<e.$o(this.i,n);L++)b[e._o(this.i,n,L)]=true;D=true;}}while(D);for(M=0;M<this.i.f&&h[M];)++M;if(M!=this.i.f){for(o=j(l.j7,H,6,this.i.f,15,1),o[0]=M,g=j(l.j7,H,6,this.i.f,15,1),g[0]=-1,u=j(l.j7,H,6,this.i.f,15,1),u[M]=1,i=0,Q=0,c=1;i<=Q;){for(L=0;L<e.$o(this.i,o[i]);L++)if(r=e.Zo(this.i,o[i],L),r!=g[o[i]]){if(u[r]!=0){e.$s(this,e._o(this.i,o[i],L),h);continue;}h[r]||(u[r]=c,g[r]=o[i],o[++Q]=r);}if(++i,i>Q){for(n=0;n<this.i.f;n++)if(u[n]==0&&!h[n]){u[n]=++c,o[++Q]=n,g[n]=-1;break;}}}if((A&4)!=0&&(this.e=j(l.Feb,ht,6,this.j.a.length,16,1),this.f=j(l.Feb,ht,6,this.j.a.length,16,1),this.d=j(l.j7,H,6,this.j.a.length,15,1),e.at(this,this.e,this.f,this.d,(A&8)!=0),e.At(this)),(A&2)!=0)for(P=0;P<this.i.g;P++)!b[P]&&e.fm(this.i,P)!=0&&(k=e.bt(this,P,h),k!=null&&e.Bt(this,k,e.jt(this,k)));}},e.Ft=function(t){return t==5||t==6||t==7||t==8||t==15||t==16||t==33||t==34;},N(151,1,{},e.Dt),U.g=0,l.X7=O(151),e.Gt=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;if(A==2||A==5)y.sj(t.F,e.ku(t.G,t.G.length)),t.b&&y.sj(t.a,e.lu(t.C,t.C.length));else if(A==4)h=e.St(e.ku(t.G,t.G.length)),y.Uwb(t.O,h)||(y.Swb(t.O,h),y.sj(t.F,e.ku(t.G,t.G.length)),t.b&&y.sj(t.a,e.lu(t.C,t.C.length)));else if(A==3){if(h=e.St(e.ku(t.G,t.G.length)),!y.Uwb(t.O,h)){for(s=false,Q=(u=new y.pwb(new y.vwb(new y.Cpb(t.O.a).a).b),new y.Lpb(u));y.kob(Q.a.a);){for(o=(D=y.nwb(Q.a),D.gd()),L=0,P=h,r=0,i=P.length;r<i;++r){for(n=P[r];L<o.length&&o[L]<n;)++L;if(L<o.length&&n==o[L]){s=true;break;}}if(s)break;}s||(y.Swb(t.O,h),y.sj(t.F,e.ku(t.G,t.G.length)),t.b&&y.sj(t.a,e.lu(t.C,t.C.length)));}}else A==6&&(h=e.Tt(t,e.ku(t.G,t.G.length)),y.Uwb(t.O,h)||(y.Swb(t.O,h),y.sj(t.F,e.ku(t.G,t.G.length)),t.b&&y.sj(t.a,e.lu(t.C,t.C.length))));},e.Ht=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b;if(u=e.$o(t.H,A),P=t.o[n],P>u)return  false;if(Q=e.Ol(t.H,A),i=e.Ol(t.g,n),r=e.Jl(t.g,n),o=e.Jl(t.H,A),pt(J(i,1),0)){if(r!=null){if(pt(J(Q,1),0)){if(o==null||!e.Yt(r,o))return  false;}else if(o!=null){if(e.$t(o,r))return  false;}else if(e.Xt(e.Tl(t.H,A),r))return  false;}}else {if(pt(J(Q,1),0))return  false;if(r!=null){if(o!=null){if(!e.Yt(o,r))return  false;}else if(!e.Xt(e.Tl(t.H,A),r))return  false;}else if(o!=null||t.J[A]!=t.k[n])return  false;}if(pt(rt(Q,i),0)){if(pt(J(i,j1),0)){if(t.H.K&&H1(J(Q,j1),0))return  false;if(P!=u)return  false;}if(pt(J(i,Z1),0)&&P>=u&&H1(J(Q,Z1),0))return  false;}if(pt(J(t.I[A],tA(t.j[n])),0)||pt(J(t.B[n],tA(t.M[A])),0))return  false;if(D=J(i,EA),t.H.K){if(L=J(i,EA),et(L,0)!=0&&(et(D,0)==0||pt(J(D,tA(L)),0)))return  false;}else if(et(D,0)!=0&&H1(J(D,t.M[A]),0))return  false;return e.Bl(t.g,n)!=0&&e.Bl(t.g,n)!=e.Bl(t.H,A)||e.Ml(t.g,n)!=0&&e.Ml(t.g,n)!=e.Ml(t.H,A)||e.Pl(t.g,n)!=0&&e.Pl(t.g,n)!=e.Pl(t.H,A)||(s=J(i,ur),et(s,0)!=0&&e.xp(t.H,A)!=Jt(t0(s,47))-7)?false:(h=J(i,kA),et(h,0)!=0?(h=t0(h,22),t.H.K?(b=Jt(t0(J(e.Ol(t.H,A),kA),22)),Jt(h)==b):e.Uo(t.H,A)==Jt(h)):true);},e.It=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h;if(Q=t.K[A],D=t.n[n],(e.hm(t.g,n)&Le)!=0){if(o=e.jm(t.H,A),r=e.jm(t.g,n),i=e.hm(t.g,n)&127,o!=r&&!(o==1&&(i&1)!=0)&&!(o==2&&(i&2)!=0)&&!(o==4&&(i&4)!=0)&&!(o==32&&(i&32)!=0)&&!(o==64&&(i&64)!=0)&&!(o==16&&(i&16)!=0)&&!(o==8&&(i&8)!=0))return  false;Q&=-128,D&=-128;}if((Q&~D)!=0)return  false;if(h=(e.hm(t.g,n)&iA)>>17,h!=0){if(t.H.K&&h==(e.hm(t.H,n)&iA)>>17)return  true;if(h<=2)return L=e.Yo(t.H,A),h==1?L>=8&&L<=12:L>=12;for(P=false,s=e.Ep(t.H),u=0;u<s.j.a.length;u++)if(y.xj(s.k,u).length==h&&e.rt(s,u,A)){P=true;break;}if(!P)return  false;}return  true;},e.Jt=function(t){var A,n;for(t.b=null,A=0;A<t.g.g;A++)e.Cm(t.g,A)&&(!t.b&&(t.b=new y.Mj()),n=new e.mu(),n.a=e.Zl(t.g,0,A),n.b=e.Zl(t.g,1,A),n.d=e._l(t.g,A),n.c=e.$l(t.g,A),y.sj(t.b,n));},e.Kt=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h;for(e.bw(t.g,t.N),o=d.Math.max(t.g.f,t.g.g)+16,t.s=j(l.j7,H,6,o,15,1),t.u=j(l.j7,H,6,o,15,1),t.v=j(l.j7,H,6,o,15,1),t.t=j(l.Feb,ht,6,o+1,16,1),D=j(l.Feb,ht,6,t.g.f,16,1),u=j(l.Feb,ht,6,t.g.g,16,1),P=0,n=0;n<t.g.f;n++)if(!t.D[n]&&!D[n])for(t.s[P]=n,t.v[P]=-1,t.u[P]=-1,Q=P;P<=Q;){for(L=0;L<e.Mo(t.g,t.s[P]);L++)Q=e.hu(t,P,Q,L,D,u,-1);for(;t.t[++P];);}if(t.w=P,t.p!=0){for(Q=t.w-1,i=0;i<t.d;i++)for(P=0;P<=Q;){for(s=0;s<e.Mo(t.g,t.s[P]);s++)Q=e.hu(t,P,Q,s,D,u,i);for(;t.t[++P];);}for(A=0;A<t.g.f;A++)if(t.D[A]&&!D[A])for(t.s[P]=A,t.v[P]=-1,t.u[P]=-1,Q=P;P<=Q;){for(s=0;s<e.Mo(t.g,t.s[P]);s++)e.Zo(t.g,t.s[P],s)<t.g.f&&(Q=e.hu(t,P,Q,s,D,u,t.f[A]));for(;t.t[++P];);}for(t.e=j(l.j7,H,6,t.d,15,1),h=0;h<t.d;h++)t.e[h]=-1;for(L=t.w;L<P;L++)r=t.f[t.s[L]],t.e[r]==-1&&(t.e[r]=L);}t.A=P;},e.Lt=function(t,A){var n,P,r,i,D,u,o,Q;for(Q=e.Ep(t.H),u=0;u<Q.j.a.length;u++)if(e.rt(Q,u,A)){for(o=j(l.j7,H,6,2,15,1),r=0;r<2;r++)for(o[r]=St,P=e.Zl(t.H,r,A),i=0;i<e.$o(t.H,P);i++)n=e.Zo(t.H,P,i),n!=e.Zl(t.H,1-r,A)&&o[r]>n&&(o[r]=n);return D=0,e.qt(Q,u,o[0])&&++D,e.qt(Q,u,o[1])&&++D,D==2?2:D==1?1:2;}return 0;},e.Mt=function(t,A,n){var P,r,i,D,u;if(t.b){for(t.C=j(l.Feb,ht,6,A.length,16,1),r=new y.Bqb(t.b);r.a<r.c.a.length;)if(P=y.Aqb(r),t.f==null||n==-1&&t.f[P.a]==-1&&t.f[P.b]==-1||n!=-1&&(t.f[P.a]==n||t.f[P.b]==n)){if(u=j(l.j7,H,6,P.c+2,15,1),i=e.yp(t.H,u,t.G[P.a],t.G[P.b],P.c+1,A,null)-1,i<P.d||i>P.c)return  false;for(D=1;D<=i;D++)t.C[u[D]]=true;}}return  true;},e.Nt=function(t,A){var n,P,r,i,D,u,o,Q;for(r=0;r<t.g.g;r++)if((e.hm(t.g,r)&hA)!=0){if(i=e.gm(t.g,r),i==0)continue;if(n=e.Zl(t.g,0,r),P=e.Zl(t.g,1,r),t.f==null||A==-1&&t.f[n]==-1&&t.f[P]==-1||A!=-1&&(t.f[n]==A||t.f[P]==A)){if(D=t.G[n],u=t.G[P],o=e.Xo(t.H,D,u),Q=e.gm(t.H,o),Q==0&&(e.jq(t.H,o)&&(Q=e.Lt(t,o)),Q==0)||i==3||Q==3)continue;if(e.Ut(t,r)==(i==Q))return  false;}}return  true;},e.Ot=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g;for(D=0,o=0;o<t.g.f;o++)if((t.f==null||t.f[o]==A)&&pt(J(e.Ol(t.g,o),L0),0)){if(h=t.G[o],s=e.Nl(t.g,o),c=e.Nl(t.H,h),s==0||s==3)continue;if(c==0||c==3)return  false;if(e.Hl(t.g,o)==1){++D;continue;}if(e.Hl(t.H,h)==1)return  false;if(e.Hl(t.g,o)==2){++D;continue;}if(e.Hl(t.H,h)==2||e.Zt(t,o)==(s==c))return  false;}if(D!=0){for(r=j(l.j7,H,6,D,15,1),i=0,Q=0;Q<t.g.f;Q++)(t.f==null||t.f[Q]==A)&&pt(J(e.Ol(t.g,Q),L0),0)&&(s=e.Nl(t.g,Q),s!=0&&s!=3&&(r[i++]=e.Gl(t.g,Q)<<24|e.Hl(t.g,Q)<<22|Q));for(w.vzb(r,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),i=0;i<r.length;)for(L=r[i]&_1,b=t.G[L],n=r[i]&-4194304,P=e.Zt(t,L)^e.Nl(t.g,L)==e.Nl(t.H,b),++i;i<r.length&&(r[i]&-4194304)==n;i++)if(u=r[i]&_1,h=t.G[u],e.Hl(t.H,h)!=e.Hl(t.H,b)||e.Gl(t.H,h)!=e.Gl(t.H,b)||(g=e.Zt(t,u)^e.Nl(t.g,u)==e.Nl(t.H,h),g!=P))return  false;}return  true;},e.Pt=function(t,A,n){var P,r,i,D,u,o,Q;if(t.P=false,t.F=new y.Mj(),t.a=new y.Mj(),y.Svb(t.O.a),!t.H||!t.g||t.g.f-t.p>t.H.f||t.g.g-t.q>t.H.g||t.g.f==0)return 0;for(A==6&&(t.N=63),e.du(t,n),P=j(l.Feb,ht,6,t.H.f,16,1),t.G=j(l.j7,H,6,t.g.f,15,1),y.Nqb(t.G,-1),u=j(l.j7,H,6,t.A,15,1),y.Tqb(u,u.length,-1),i=0;!t.P;){if(t.w!=0){if(Q=t.u[i]==-1?t.H.f:e.Mo(t.H,t.G[t.u[i]]),++u[i],u[i]==Q){if(u[i]=-1,i==0)break;--i,t.t[i]||(P[t.G[t.s[i]]]=false);continue;}if(t.u[i]==-1)P[u[i]]||e.Ht(t,u[i],t.s[i])&&(t.G[t.s[i]]=u[i],P[u[i]]=true,++i);else {if(e.Zo(t.H,t.G[t.u[i]],u[i])>=t.H.f)continue;r=e.Zo(t.H,t.G[t.u[i]],u[i]),t.t[i]?r==t.G[t.s[i]]&&e.It(t,e._o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&++i:P[r]||e.Ht(t,r,t.s[i])&&e.It(t,e._o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&(P[r]=true,t.G[t.s[i]]=r,++i);}}if(i==t.w){if(e.Ot(t,-1)&&e.Nt(t,-1)&&e.Mt(t,P,-1)){for(o=false,D=0;D<t.d;D++)if(e.Vt(t,P,u,D)){o=true;break;}if(A==1&&!o||!o&&(e.Gt(t,A),A==2))return 1;}if(i==0)break;--i,t.t[i]||(P[t.G[t.s[i]]]=false);}}return t.F.a.length;},e.Qt=function(t,A){var n,P,r,i,D,u,o,Q;if(o=0,t.K){switch(A<t.f&&e.vt(t.p,A)?o=rt(o,Ur):A<t.f&&e.ot(t.p,A)&&(o=rt(o,2)),Q=(r=t.u[A]&V0,r==0?0:r==i0?2:r==j1?3:4),Q!=0&&(o=rt(o,8),Q>2&&(o=rt(o,16)),Q>3&&(o=rt(o,32))),n=t.s[A],n<0?o=rt(o,oe):n>0&&(o=rt(o,ue)),D=t.j[A],D){case 0:break;case 1:o=rt(o,n0);break;case 2:o=rt(o,te);break;case 3:o=rt(o,iA);break;default:o=rt(o,le);}switch(P=e.Po(t,A),P){case 0:break;case 1:o=rt(o,Ce);break;case 2:o=rt(o,Gn);break;case 3:o=rt(o,Jn);break;default:o=rt(o,Dn);}u=t.o[A],u>0&&(o=rt(o,I0)),u>1&&(o=rt(o,LA));}else {switch(A<t.f&&e.vt(t.p,A)?o=rt(o,Ur):A<t.f&&e.ot(t.p,A)?o=rt(o,2):o=rt(o,4),(t.u[A]&S0)!=0?o=rt(o,tn):o=rt(o,An),Q=(r=t.u[A]&V0,r==0?0:r==i0?2:r==j1?3:4),Q==0?o=rt(o,112):Q==2?o=rt(o,104):Q==3?o=rt(o,88):o=rt(o,56),n=t.s[A],n==0?o=rt(o,nn):n<0?o=rt(o,oe):o=rt(o,ue),i=t.e[A]-e.tp(t,A)+e.op(t,A),i){case 0:o=rt(o,1792);break;case 1:o=rt(o,1664);break;case 2:o=rt(o,1408);break;default:o=rt(o,896);}switch(D=t.j[A],D){case 0:o=rt(o,3932160);break;case 1:o=rt(o,Pn);break;case 2:o=rt(o,pe);break;case 3:o=rt(o,Re);break;default:o=rt(o,le);}switch(P=e.Po(t,A),P){case 0:o=rt(o,zn);break;case 1:o=rt(o,Nn);break;case 2:o=rt(o,In);break;case 3:o=rt(o,On);break;default:o=rt(o,Dn);}switch(u=t.o[A],u){case 0:o=rt(o,98304);break;case 1:o=rt(o,81920);break;default:o=rt(o,49152);}}return o;},e.Rt=function(t,A){var n;if(n=0,A<t.g&&(e.ut(t.p,A)||t.H[A]==8)||t.H[A]==8)n|=8;else switch(e.fm(t,A)){case 0:n|=16;break;case 1:n|=1;break;case 2:n|=2;break;case 3:n|=4;break;case 4:n|=32;break;case 5:n|=64;}return (t.F[A]&64)!=0?n|=256:t.K||(n|=128),A<t.g&&e.pt(t.p,A)?n|=v0:t.K||(n|=_0),n;},e.St=function(t){var A,n,P,r,i,D,u,o,Q,L;for(o=0,P=t,i=0,u=P.length;i<u;++i)A=P[i],A==-1&&++o;if(o!=0)for(L=t,t=j(l.j7,H,6,L.length-o,15,1),Q=0,n=L,r=0,D=n.length;r<D;++r)A=n[r],A!=-1&&(t[Q++]=A);return w.vzb(t,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),t;},e.Tt=function(t,A){var n,P,r,i,D,u,o,Q;for(D=0,P=A,r=0,i=P.length;r<i;++r)n=P[r],n==-1&&++D;for(Q=j(l.j7,H,6,A.length-D,15,1),o=0,u=0;u<A.length;u++)A[u]!=-1&&(Q[o]=e.mw(t.g,u)<<16|e.mw(t.H,A[u]),t.i!=null&&(Q[o]|=t.i[u]<<24),++o);return w.vzb(Q,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),Q;},e.Ut=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b;for(i=false,r=0;r<2;r++)if(n=e.Zl(t.g,r,A),L=t.G[n],e.$o(t.H,L)>2){for(h=e.Zl(t.g,1-r,A),o=St,u=0;u<e.$o(t.g,n);u++)P=e.Zo(t.g,n,u),P!=h&&o>P&&(o=P);for(b=t.G[h],Q=St,D=0;D<e.$o(t.H,L);D++)s=e.Zo(t.H,L,D),s!=b&&Q>s&&(Q=s);t.G[o]!=Q&&(i=!i);}return i;},e.Vt=function(t,A,n,P){var r,i,D,u,o,Q,L,s;for(u=t.e[P],o=u+1;o<t.A&&t.f[t.s[o]]==P;)++o;for(L=u;L<o;L++)n[L]=-1;for(D=u;;){if(s=t.u[D]==-1?t.H.f:e.Mo(t.H,t.G[t.u[D]]),++n[D],n[D]==s){if(n[D]=-1,D==u)break;--D,t.t[D]||(A[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);continue;}if(t.u[D]==-1)A[n[D]]||e.Ht(t,n[D],t.s[D])&&(t.G[t.s[D]]=n[D],A[n[D]]=true,++D);else {if(e.Zo(t.H,t.G[t.u[D]],n[D])>=t.H.f)continue;i=e.Zo(t.H,t.G[t.u[D]],n[D]),t.t[D]?i==t.G[t.s[D]]&&e.It(t,e._o(t.H,t.G[t.u[D]],n[D]),t.v[D])&&++D:A[i]||e.Ht(t,i,t.s[D])&&e.It(t,e._o(t.H,t.G[t.u[D]],n[D]),t.v[D])&&(A[i]=true,t.G[t.s[D]]=i,++D);}if(D==o){if(e.Ot(t,P)&&e.Nt(t,P)&&e.Mt(t,A,P)){for(Q=u;Q<o;Q++)t.t[Q]||(r=t.s[Q],A[t.G[r]]=false,t.G[r]=-1);return  true;}--D,t.t[D]||(A[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);}}return  false;},e.Wt=function(t){return e.Pt(t,1,t.c)>0;},e.Xt=function(t,A){var n;for(n=0;n<A.length;n++)if(A[n]==t)return  true;return  false;},e.Yt=function(t,A){var n,P,r;for(r=0,P=0;P<t.length;P++){for(n=t[P];A[r]<n;)if(++r,r==A.length)return  false;if(A[r]>n)return  false;}return  true;},e.Zt=function(t,A){var n,P,r,i,D,u,o,Q;if(D=false,e.Qo(t.g,A)==0)for(i=1;i<e.$o(t.g,A);i++)for(u=0;u<i;u++)P=e.Zo(t.g,A,i),r=e.Zo(t.g,A,u),t.G[P]>t.G[r]^P>r&&(D=!D);else for(i=0;i<e.$o(t.g,A);i++){for(n=e.Zo(t.g,A,i),Q=0,o=j(l.j7,H,6,3,15,1),u=0;u<e.$o(t.g,n);u++)o[Q]=e.Zo(t.g,n,u),o[Q]!=A&&++Q;Q==2&&t.G[o[0]]>t.G[o[1]]^o[0]>o[1]&&(D=!D);}return D;},e.$t=function(t,A){var n,P,r,i;for(r=0,i=0;r<t.length&&i<A.length;){if(n=t[r],P=A[i],n==P)return  true;n<P?++r:++i;}return  false;},e._t=function(t,A){var n,P,r,i,D;if(!A||A.q==0||!A.K){t.g=null;return;}for(t.g=A,t.r=false,e.bw(t.g,1),t.N=7,P=0;P<t.g.f;P++)pt(J(e.Ol(t.g,P),{l:L0,m:0,h:3}),0)&&(t.N=15);for(D=0;D<t.g.g;D++)(e.hm(t.g,D)&hA)!=0&&(t.N=15);for(t.L&&t.N!=7&&e.bw(t.H,t.N),t.p=0,t.q=0,t.D=j(l.Feb,ht,6,t.g.f,16,1),r=0;r<t.g.f;r++)t.D[r]=e.Jm(t.g,r),t.D[r]&&++t.p;if(t.d=0,t.f=null,t.i=null,t.p!=0){for(i=0;i<t.g.g;i++)(t.D[e.Zl(t.g,0,i)]||t.D[e.Zl(t.g,1,i)])&&++t.q;for(n=0;n<t.g.q;n++)e.sn(t.g,n,t.D[n]);t.f=j(l.j7,H,6,t.g.q,15,1),t.d=e.hp(t.g,t.f,true,false);}},e.au=function(t,A){t.i=A;},e.bu=function(t,A,n){e.cu(t,n),e._t(t,A);},e.cu=function(t,A){if(!A||A.q==0){t.H=null;return;}t.H=A,t.L=false,e.bw(t.H,1);},e.du=function(t,A){t.L||(e.gu(t,A),t.L=true),t.r||(e.eu(t,A),e.Kt(t),e.Jt(t),t.r=true);},e.eu=function(t,A){var n,P,r,i,D,u,o,Q,L,s;for(i=null,o=null,D=null,e.bw(t.g,t.N),t.o=j(l.j7,H,6,t.g.f,15,1),P=0;P<t.g.f;P++)t.o[P]=e.$o(t.g,P);if(t.p!=0){for(Q=new e.uw(t.g.q,t.g.r),s=j(l.Feb,ht,6,t.g.q,16,1),r=0;r<t.g.q;r++)s[r]=!t.D[r];for(e.Co(t.g,Q,s,true,null),e.bw(Q,t.N),e.fu(t,Q,A),i=t.j,o=t.n,D=t.k,L=0,n=0;n<t.g.f;n++)t.D[n]||(t.o[n]=e.$o(Q,L++));}if(e.fu(t,t.g,A),t.p!=0){for(L=0,n=0;n<t.g.q;n++)t.D[n]||(t.j[n]=i[L],t.k[n]=D[L++]);for(L=0,u=0;u<t.g.r;u++)!t.D[e.Zl(t.g,0,u)]&&!t.D[e.Zl(t.g,1,u)]&&(t.n[u]=o[L++]);}},e.fu=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;for(g=A.f,t.j=j(l.k7,e0,6,A.f,14,1),t.k=j(l.j7,H,6,A.f,15,1),u=0;u<g;u++)t.j[u]=ne(J(rt(e.Qt(A,u),A.B[u]),yr),W8),t.k[u]=A.C[u],(n&1)!=0&&(t.k[u]+=A.s[u]+16<<8),(n&2)!=0&&(t.k[u]+=A.A[u]<<16);for(t.B=j(l.k7,e0,6,A.f,14,1),k=(e.bw(A,7),A.p),c=0;c<k.j.a.length;c++){for(b=false,Q=y.xj(k.k,c),L=0,s=Q.length;L<s;++L)if(h=Q[L],(A.G[h]&UA)!=0){b=true;break;}if(!b)for(M=y.xj(k.k,c).length,r=y.xj(k.j,c),i=0,D=r.length;i<D;++i)P=r[i],M==3?t.B[P]=rt(t.B[P],SA):M==4?t.B[P]=rt(t.B[P],VA):M==5?t.B[P]=rt(t.B[P],xA):M==6?t.B[P]=rt(t.B[P],zA):M==7&&(t.B[P]=rt(t.B[P],NA));}for(B=A.g,t.n=j(l.j7,H,6,A.g,15,1),o=0;o<B;o++)t.n[o]=(e.Rt(A,o)|A.G[o])&6291967^6291840,(n&4)!=0?(t.n[o]&2)!=0&&(t.n[o]|=8):(n&8)!=0&&(t.n[o]&2)!=0&&o<A.g&&e.pt(A.p,o)&&(t.n[o]|=8);},e.gu=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b;for(e.bw(t.H,t.N),L=t.H.f,t.J=j(l.j7,H,6,L,15,1),t.I=j(l.k7,e0,6,L,14,1),D=0;D<L;D++)t.I[D]=ne(J(rt(e.Qt(t.H,D),e.Ol(t.H,D)),yr),W8),t.J[D]=e.Tl(t.H,D),(A&1)!=0&&(t.J[D]+=e.Bl(t.H,D)+16<<8),(A&2)!=0&&(t.J[D]+=e.Ml(t.H,D)<<16);for(t.M=j(l.k7,e0,6,L,14,1),h=e.Ep(t.H),Q=0;Q<h.j.a.length;Q++)for(b=y.xj(h.k,Q).length,P=y.xj(h.j,Q),r=0,i=P.length;r<i;++r)u=P[r],b==3?t.M[u]=rt(t.M[u],SA):b==4?t.M[u]=rt(t.M[u],VA):b==5?t.M[u]=rt(t.M[u],xA):b==6?t.M[u]=rt(t.M[u],zA):b==7&&(t.M[u]=rt(t.M[u],NA));for(n=0;n<L;n++)b=e.Uo(t.H,n),b==0?t.M[n]=rt(t.M[n],QA):b>7&&(t.M[n]=rt(t.M[n],De));for(s=t.H.g,t.K=j(l.j7,H,6,s,15,1),o=0;o<s;o++)t.K[o]=(e.Rt(t.H,o)|e.hm(t.H,o))&6422527^6291840;},e.hu=function(t,A,n,P,r,i,D){var u,o;return u=e.Zo(t.g,t.s[A],P),u<t.D.length&&(!t.D[u]||t.f[u]==D)&&u!=t.u[A]&&(o=e._o(t.g,t.s[A],P),!i[o]&&!e.Cm(t.g,o)&&(t.s[++n]=u,t.u[n]=t.s[A],t.v[n]=o,i[o]=true,r[u]?t.t[n]=true:r[u]=true)),n;},e.iu=function(){this.c=8,this.O=new y.Ywb(new Lt._3());},e.ju=function(t){this.c=t,this.O=new y.Ywb(new Lt._3());},e.ku=function(t,A){var n;return n=j(l.j7,H,6,A,15,1),T.$jb(t,0,n,0,d.Math.min(t.length,A)),n;},e.lu=function(t,A){var n;return n=j(l.Feb,ht,6,A,16,1),T.$jb(t,0,n,0,d.Math.min(t.length,A)),n;},N(64,1,{},e.iu,e.ju),U.c=0,U.d=0,U.p=0,U.q=0,U.r=false,U.w=0,U.A=0,U.L=false,U.N=0,U.P=false,l.$7=O(64),e.mu=function(){},N(215,1,{215:1},e.mu),U.a=0,U.b=0,U.c=0,U.d=0,l.Y7=O(215),e.pu=function(){e.pu=D1,e.nu=v(F(l.Ybb,1),st,2,6,["QM@HzAmdqjF@","RF@Q``","qC`@ISTAlQE`","`J@H","QM@HzAmdqbF@","qC`@ISTAlQEhqPp@","sJP@DiZhAmQEb","RF@QPvR@","QM@HzA@","qC`@ISTAlQEhpPp@","qC`@Qz`MbHl","sJP@DiZhAmQEcFZF@","RFPDXH","qC`@IVtAlQE`","QM@HvAmdqfF@","sGP@DiVj`FsDVM@","`L@H","sJP@DizhAmQEcFBF@","sJP@DjvhAmQEb","sFp@DiTt@@AlqEcP","sGP@LdbMU@MfHlZ","QMHAIhD","QM@HzAy@","sJP@DkVhAmQEb","sNp@DiUjj@[\\QXu`","sJP@DiZhAmQEcFBF@","sGP@DjVj`FsDVM@","RFPDTH","RG@DXOH@","sGP@Divj`FsDVMcAC@","sGP@Dj}j`FsDVM@","qC`@Qz`MbHmFRF@","sNp@LdbJjj@[\\QXu`","QMHAIhGe@","QM@HzAyd`","QM`AIhD","qC`@ISTA@","sGP@DkUj`FsDVM@","qC`@IVtAlQEhqPp@","sNp@DiUjj@[\\QXuqea`@","KAx@@IRjuUPAlHPfES\\","QM`BN`P","sJP@DjZhAmQEcFJF@","Hid@@DjU^nBBH@FtaBXUMp`","sNp@Diujj@[\\QXuq`a`@","sJP@DjvhAmQEcFZF@","sJP@DjZhAmQEcFFF@","sOp@DjWkB@@FwDVM\\YhX@","sNp@Dj}Zj@[\\QXu`","sNp@DiWjj@[\\QXuq`a`@","sOp@DjWkB@@D","KAx@@ITouUPAlHPfES\\","KAx@@YIDTjjh@vDHSBin@","sNp@DkUZj@[\\QXu`","RFPDXOH@","QM`BN`^L`","qC`@ISTAy@","sGP@LdbMU@MfHl[FVF@","qCb@AIZ`H","KAx@@IRjuUPAlHPfES]FFa`@","KAx@@ITnuUPAlHPfES\\","HiD@@DiUVjj`AmHPfES\\H","sNp@DjUjj@[\\QXu`","sJP@DkVhAmQEcFJF@","sGP@DjVj`FsDVMcCC@","qC`@Qz`MbHmFBF@","sJP@DkfhAmQEb","qC`@IVtAlQEhsPp@","sGP@Djuj`FsDVM@","sGP@Dj}j`FsDVMcMC@","sJP@DiZhA@","KAx@@ISjuUPAlHPfES]F@a`@","sJP@DjZhAmQEcFRF@","KAx@@IRnuUPAlHPfES]F@a`@","HiD@@DjWvjj`AmHPfES\\H","QMHAIhGd@","sNp@DiUjj@[\\QXuq`a`@","KAx@@IVjmUPAlHPfES\\","sGP@DjVj`FsDVMcMC@","QM`AIhGe@","HiD@@LdbJRjjh@[RDIaTwB","qCp@AIZ`H","sGP@LdbMU@MfHl[FFF@","QMDARVA@","sNp@LdbJjj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqca`@","sGP@Dkej`FsDVM@","qCb@AIZ`OI@","HaD@@DjUZxHH@AlHPfES]FLa`@","sGP@DkYj`FsDVM@","qCb@AIV`H","sNp@LdbJjj@[\\QXuqea`@","sGP@DkUj`FsDVMcEC@","sFp@DiTt@@Axa@","Hmt@@DjU_ZxHHj@AmhPfES\\Lj","QM`BN`^P","qCb@AIZ`OH`","sFp@DiTt@@AxaP","sGP@Djuj`FsDVMcEC@","sGP@Djuj`FsDVMcIC@","sGP@DkUj`FsDVMcKC@","sJP@DkfhAmQEcFRF@","sGP@DjVj`FsDVMcIC@","HaD@@DjUZxHH@AlHPfES]FFa`@","qC`@IRtDVqDV@","sNp@Dj}Zj@[\\QXuqfa`@","KAx@@ITnuUPAlHPfES]FFa`@","HiD@@DkUUjj`AmHPfES\\H","sJQ@@dkU@H","qC`@Qz`H","KAx@@IUkmUPAlHPfES\\","KAx@@ITouUPAlHPfES]FJa`@","sJP@H~j@[TQX`","sGP@DjZj`FsDVM@","sJP@DkVhAmQEcFFF@","sJX@@eKU@H","sJP@DizhAy@","QMHAIhGbP","KAx@@ITouUPAlHPfES]FNa`@","HaD@@DjUZxHD@AlHPfES\\","HaD@@DjUZxHH@A@","sNp@LdbJjj@[\\QXuqaa`@","Hed@@LdbRQUUUP@vTHSBinFP","KAx@@ITouUPAlHPfES]FLa`@","sNp@DkUZj@[\\QXuqba`@","KAx@@ITjuUPAlHPfES]FNa`@","KAx@@YIDTjjh@vDHSBincGPp@","HaD@@DjYvxH`@AlHPfES]FLa`@","RF@QP`","qCb@AIj`H","sNp@DjUjj@[\\QXuqaa`@","sNp@DkVZj@[\\QXu`","KAx@@YIDUJjh@vDHSBin@","sGP@DkYj`FsDVMcIC@","sGP@DjVj`FsDVMcAC@","sGP@DiVj`D","sJP@DkVhAmQEcFZF@","sNp@LdbLjj@[\\QXu`","QM@HvAmdqbF@","HaD@@DjWjXHB@AlHPfES\\","sNp@DjwZj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqda`@","sFp@DiTt@@Axa`","HiD@@Djuujj`AmHPfES\\H","sNp@DkUZj@[\\QXuqca`@","sJP@DiZhAy@","KAx@@YIDTjjh@vDHSBincCPp@","KAx@@IWNmUPAlHPfES\\","KAx@@IVkMUPAlHPfES\\","sJQ@@dju@H","qCb@AIZ`OH@","qC`@ISTAxa@","sNp@DjyZj@[\\QXu`","Hid@@DjUfaBB`@FtaBXUMp`","HiD@@DiUVjj`AmHPfES\\LXBF@","KAx@@IUjmUPAlHPfES\\","HiD@@DjWvjj`AmHPfES\\LXjF@","sJP@DjVhAmQEb","qCb@AIV`OH`","HiD@@LdbJRjjh@[RDIaTwCFDa`@","KAx@@YIDTjjh@vDHSBinc@Pp@","sNp@DjUjj@[\\QXuqda`@","qC`@Qz`OED","sJP@DkfhAmQEcFZF@","KAx@@YIDbjjh@vDHSBincDPp@","sGP@Djyj`FsDVMcMC@","KAx@@IVrmUPAlHPfES\\","qCp@AIZ`OI@","sJX@@dkU@H","sJQ@@dkU@OH`","sNp@Di]ZjBBvxbqk@","Hkl@@DjU_Uk``bj`@[VDIaTwCJzX","sGP@DjZj`FsDVMcEC@","Hid@@DjU^nBBH@FtaBXUMpqcHX@","sNp@DkeZj@[\\QXu`","sNp@DjYjj@[\\QXuqca`@","sGQ@@djuT@`","HiD@@LdbJTjjh@[RDIaTwB","sOp@DjWkB@@Gd`","HeT@@LdbbRKBDQD@CYPaLJfxY@","qCr@XIKTA@","HiD@@DjW^jj`AmHPfES\\LXJF@","HeT@@DjU]k``b`@[JDIaTwCH","sGP@Djuj`FsDVMcCC@","`IH`B","sOp@DjWkB@@GdX","sJQ@@eKU@H","KAx@@YIDUJjh@vDHSBincBPp@","sJX@@eKU@OH@","KAx@@YIDTjjh@vDHSBincAPp@","sOq@@drm\\@@@`","KAx@@IUkMUPAlHPfES\\","qCp@AIj`H","Hed@@DjUUjjj@FraBXUMpr","sGX@@eJuT@`","sGP@DkUj`FsDVMcCC@","HiD@@Dj}Ujj`AmHPfES\\LXrF@","KAx@@ITouUPAlHPfES]FHa`@","Hed@@DjWujjj@FraBXUMpsFIa`@","sGP@DiUj``mfHlZ","sFp@DiTvjhAlqEcP","Hid@@DjU^nBBH@FtaBXUMpq`XX@","sJP@DkVdAmQEb","qCp@AIZ`OH`","QMhDRVA@","qC`@ISJAlQE`","qCp@BOTAyhl","sJX@@eOU@ODB","sFp@DiTt@@AyaB","sGP@DkUj`FsDVMcMC@","Hid@@DjYUaBH`@FtaBXUMpqcHX@","qC`@Qz`OH@","HiD@@DjUVjj`AmHPfES\\LXZF@","sJP@H~j@[TQXqda`@","sJX@@eKU@OI@","sNp@Djejj@[\\QXu`","sJQ@@dsU@H","sJQ@@dkU@OI`","KAx@@YIMDVjh@vDHSBin@","Hid@@DjU^nBBD@FtaBXUMp`","sNp@DkgZj@[\\QXuqca`@","qC`@IRtDVqDVcEC@","Hed@@LdbRQeUUP@vTHSBinFP","sNp@DiUjj@P","qC`@IRtDT","sNp@DkYZj@[\\QXuqca`@","KAx@@IUkmUPAlHPfES]FDa`@","KAx@@IVjmUPAlHPfES]FNa`@","sOx@@drm\\@@@`","KAx@@ITjuUPAlHPfES]FBa`@","QMDARVAyH","sJP`@dfvhA@","HeT@@DjU_k``b`@[JDIaTwCLXfF@","KAx@@IToUUPAlHPfES]FJa`@","sGP@DkYj`FsDVMcEC@","qCb@AIZ`ODH","`I@`B","KAx@@IUzmUPAlHPfES]FFa`@","sNp@DkfZj@[\\QXu`","KAx@@ITnuUPAlHPfES]F@a`@","HiD@@LddURjjh@[RDIaTwB","sNp@Dj~Zj@[\\QXuqfa`@","Hed@@Dj{uZjj@FraBXUMpr","KAx@@ITsUUPAlHPfES\\","Hid@@LdbRQk``b@AmHPfES\\LXrF@","sOp@DjWkB@@GdH","sJQ@@dkU@OH@","Hid@@DjU^nBBH@FtaBXUMpqahX@","sGP@DiYj``mfHlZ","KAx@@IToUUPAlHPfES]FLa`@","qCp@AJZ`ODH","Hmt@@DjU]ZxHHj@AmhPfES\\Lj","sGP@DkUjPFsDVM@","qC`@IVtA@","Hed@@LdbJReUUP@vTHSBinFP","sNp@DjuZj@[\\QXuqea`@","KAx@@IUkmUPAlHPfES]FNa`@","HiD@@DkVUjj`AmHPfES\\H","Hed@@DkUeZjj@FraBXUMpr","sNp@DkVZj@[\\QXuqea`@","sJP@DiVhHKZbKFLLL@","HiD@@Djuyjj`AmHPfES\\H","sNp@DjUjj@[\\QXuq`a`@","HeT@@DjYUXPbH`@[JDIaTwCH","HiD@@DjwUjj`AmHPfES\\LXRF@","sNq@@djmUPB","KAx@@YIEEZjh@vDHSBincCPp@","sGP@Di^V`dmfHlZ","Hid@@DjYUaBHP@FtaBXUMp`","sNp@DjYjj@[\\QXuqba`@","sGP@Dkej`FsDVMcKC@","HeT@@DjU^k``b`@[JDIaTwCH","qC`@Qv`MbHmFBF@","sGQ@@djmT@`","qCr@XIKTAyH","qC`@IVtAlQEhpPp@","Hid@@LdbbQxXF@@AmHPfES\\LXjF@","sGP@DkYj`FsDVMcCC@","KAx@@IVsMUPAlHPfES\\","qCp@AIj`ODl","HiD@@DkeUjj`AmHPfES\\H","deT@@DjU[k``b`@vTHSBinFP","sJP@DkVdAmQEcFRF@","HiD@@LdbJTjjh@[RDIaTwCFDa`@","HiD@@DkYyjj`AmHPfES\\H","sJP@DjZhAyH","KAx@@IVkMUPAlHPfES]FDa`@","sJX@@dkU@OI@","Hed@@LdbRQUUUP@vTHSBinFXpLL@","Hed@@DjuUZjj@FraBXUMpr","sGP@Djfj`FsDVMcKC@","sNp@DkVZj@[\\QXuqba`@","sNp@DjyZj@[\\QXuqfa`@","qCb@AIj`OH@","sNp@DjUZj@[\\QXu`","KAx@@IWOMUPAlHPfES\\","Hid@@DjU^nBBH@D","Hed@@DjuvZjj@FraBXUMpr","sJP@DiVhHKZbKFLtL@","dmt@@DjU_ZxHHj@C[PaLJfxYVLKC@","sNp@DjuZj@[\\QXuqca`@","sJP@DkfhAmQEcFJF@","sNp@LdbJZj@[\\QXu`","HeT@@DjU_k``b`@[JDIaTwCLXFF@","KAx@@IVlmUPAlHPfES]FNa`@","HeT@@LdbbRKBDQD@CYPaLJfxYcEPp@","Hid@@DjUZnBBH@FtaBXUMpqcHX@","qCa@CIKTA@","HiD@@Dj~]jj`AmHPfES\\LXFF@","sKP@Di\\Zj@[TQX`","sGP@Djfj`FsDVMcEC@","HiD@@DkgYjj`AmHPfES\\H","sNp@DjuZj@[\\QXuqaa`@","KAx@@YIMDVjh@vDHSBincDPp@","sJP@DjVhHKZbKFLTL@","Hid@@LdbRQk``b@AmHPfES\\LXZF@","HiD@@Dj}Ujj`AmHPfES\\LXzF@","HeT@@DjU_k``bP@[JDIaTwCH","sNp@DkUZi@[\\QXu`","HiD@@DjYfjj`AmHPfES\\H","sGP@DjZj`FsDVMcAC@","Hmt@@DjU_jxHHj@AmhPfES\\Lj","Hid@@LdbRQk``R@AmHPfES\\H","KAx@@YIDUJjh@vDHSBincDPp@","qCr@XIKTAyD","sOq@@drm\\@@@|`@","Hed@@DjW^jjj@FraBXUMpsFBa`@","HeT@@DjY]zXFB@@[JDIaTwCH","Hkl@@DjU_Vk``bj`@[VDIaTwCJzX","Hid@@DjY}nBHH@FtaBXUMpqcHX@","sGX@@eKuT@|d@","sGP@Dj^Y`FsDVM@","HcL@@DjU_ZnBBJh@FqaBXUMprn`","sJP@DkVdAmQEcFJF@","sOq@@drm\\@@@|b@","sNp@DjyZj@[\\QXuqaa`@","HaD@@DjUZxHH@AyD@","qC`@Qv`H","dmt@@DjU_ZxHHj@C[PaLJfxYVLYC@","sGP@Dkej`FsDVMcMC@","Hed@@DjUUjjj@FraBXUMpsFHa`@","HeT@@LdbbRkBDQD@CYPaLJfxY@","KAx@@IU{MUPAlHPfES]FLa`@","RG@DTH","sJY@DDeVhA@","KAx@@YIDUJjh@vDHSBinc@Pp@","sJX@@dkU@OI`","sJQ@@dju@OI`","HeT@@LdbbRKBDQD@CYPaLJfxYcFPp@","sFp@DiTvjhAlqEcXpPp@","HaD@@DjUZxHH@AyG@","sNx@@eJ}UPB","sNp@LddUjj@[\\QXuqca`@","HaDH@@RVU[j@@@D","sNp@DkgZi@[\\QXu`","sGY@LDeVj`D","sNp@LdbJfZBZvxbqk@","sJP`@dfvhAyL","sGX@AddQjhAxe`","Hmt@@DjU_ZxHHj@AmhPfES\\LkFIa`@","qCh@CIKTA@","sNp@LdbLjj@[\\QXuq`a`@","sOq@@drm\\@@@|a@","KAx@@IUzmUPAlHPfES]FJa`@","sNx@AddQUUPB","sGP@Di]jP`mfHlZ","sJP`@TeZhA@","KAx@@IRjmUPHKXPaLJfx","HeT@@LdbRTM\\DDT@CYPaLJfxY@","HaF@@@Rfu[j@@@D","Hid@@DjYUaBH`@FtaBXUMpqchX@","KAx@@IUjmTpAlHPfES\\","Hid@@DjU^nBBD@FtaBXUMpqcHX@","sGP@DiUj``mfHl[FFF@","KAx@@IUvmUPAlHPfES]FLa`@","Hed@@LdbQTUUUP@vTHSBinFXqDL@","sJP@DkVhA@","sOx@@drm\\@@@|b@","KAx@@IUkMUPAlHPfES]FDa`@","HeT@@LdbRQU\\DDT@CYPaLJfxY@","HiD@@Dj}Yjj`AmHPfES\\LXrF@","HiD@@Dj{ujj`AmHPfES\\LXFF@","KAx@@IWNmUPAlHPfES]FFa`@","KAx@@IRkMUPHKXPaLJfx","sJP@DjYdAmQEcFZF@","sJY@LDeZhAyL","HaDH@@RVU[f@@@D","sJP`@deVhAyB","HaD@@DjWjZjj`AlHPfES\\","sGP@DkYj`FsDVMcMC@","sNp@DkgZj@[\\QXuqea`@","sJQ@@dlu@H","HeT@@DjU]k``b`@[JDIaTwCLXrF@","sJX@@dkU@OH`","RFDDQFCr`","sJP@DiYXIKZbKFLLL@","KAx@@YIHjjjh@vDHSBincGPp@","Hk\\@@DjU^ukmLHH@@@AmXPfES\\Lki`","sGQ@@djmT@|b@","Hid@@DjUfaBB`@FtaBXUMpqahX@","sNx@@eRmUPB","Hmt@@LdbRVak``ah@FvaBXUMprh","qCr@XIJtA@","KAx@@IWMmUPAlHPfES]FNa`@","HeT@@DjYYZPbJ@@[JDIaTwCH","sNp@DkfZj@[\\QXuqea`@","Hid@@DjU^nBAHAEVtaBXUMp`","dmt@@DjUgZDHJJ@C[PaLJfxYT","sGP@DkejPFsDVM@","sNx@@eJmUPB","qCb@AIf`H","HcL@@DjU_VnBBJh@FqaBXUMprnqcXX@","Hid@@DjUZnBBH@FtaBXUMpqahX@","sNp@LdbQZjBBvxbqkcGC@","sOx@@drm\\@@@|c@","sJP@H~j@^R@","KAx@@YIDcFjhDElHPfES\\","Hid@@DjUZnBAH@FtaBXUMp`","sNp@LddUji@[\\QXu`","sGP@DjfjPFsDVM@","HeT@@DjYUXPbD`@[JDIaTwCH","KAx@@IUoMUPAlHPfES]FDa`@","sFp@DiTt@@AyaD","Hed@@DjuuZjj@FraBXUMpsFIa`@","HeT@@DjUghP`h`@[JDIaTwCLXfF@","sOp@DjWkjj`FwDVM\\YhX@","sGP@Djfj`FsDVMcIC@","KAx@@IRkmUPHKXPaLJfzL]C@","sNx@@djmUPB","QM`AIdD","sOp@DjWkB@@Gbe@","sNp@DjyZj@[\\QXuqca`@","QM@HuAmd`","sNp@LddUjj@[\\QXuqea`@","daD@@DjUZxHD@CXPaLJfzLYC@","qCb@AIZPH","HiD@@LdbJTjjh@[RDIaTwCF@a`@","Hmt@@DjU_ZxHHi@AmhPfES\\Lj","HaDH@@RYWih@H@D","HiD@@LdbJTjjh@[RDIaTwCFHa`@","sGX@@djuT@|a@","sNp@DkfZj@[\\QXuqaa`@","Hid@@DjU^nBBH@GdL","KAx@@IVkMUPAlHPfES]FJa`@","qCr@XIKTAy@","HmT@@Dj{uVjjh@[ZDIaTwCJqaXX@","dmt@@DjYWVDHbJ@C[PaLJfxYT","Hif@@@RUe^Fh@@@P","HaDH@@Rfu[j@@@GdH","KAx@@IVsMUPAlHPfES]FDa`@","sKP@Di\\Zj@[TQXq`a`@","sJX@@eMU@OH@","HeT@@DjU^k``b`@[JDIaTwCLXFF@","Hmt@@LdbbRJXPbHh@FvaBXUMprh","sJP@DjvhAmQEcFBF@","dmt@@LdbbRNXPbHh@MmBDpj[aeXplL@","sJP`@dfvhAyD","sGP@Di^V`dmfHl[FVF@","KAx@@IVsmUPAlHPfES]FBa`@","sOq@@drm\\@@@|PP","sJY@BDeZhA@","HeT@@LdbRbmBDED@CYPaLJfxY@","Hed@@Djy[Zjj@FraBXUMpr","HeT@@DjU]k``b`@[JDIaTwCLXFF@","Hid@@DjUfaBB`@D","qCa@CIJtA@","QMPARVA@","Hid@@DjUfaBB`@FtaBXUMpqcHX@","sJY@BDfZhA@","HeT@@DjUghP`hP@[JDIaTwCH","Hed@@Dj{uZjj@FraBXUMpsFIa`@","dmt@@DjYUvDHbJ@C[PaLJfxYVLUC@","sNp`@dfuZj@P","sJQ@@dmU@OH@","sJX@@dmU@H","HeT@@DjU]k``b`@[JDIaTwCLXZF@","HiD@@LdfbJZjh@[RDIaTwCFAa`@","sOx@@drm\\@@@|a@","deT@@DjUfhP`h`@vTHSBinFP","dmt@@DjU]ZxHHj@C[PaLJfxYVLKC@","sOp@DjWkjj`FwDVM\\XHX@","HcL@@LdbbRNSBDQEP@McBDpj[ae]cFpp@","HiD@@Dj}Yji`AmHPfES\\H","HaDH@@RYe[hB@@D","Hid@@DjU^njjj@FtaBXUMpq`XX@","deT@@LdbbRKBDQB@FraBXUMpr","QMPARZA@","sOq@@drm\\@@@|QX","HaD@@DjYvxH`@A@","HcL@@LdbbRNcBDQEP@McBDpj[ae]@","QMhDRZA@","RG@DXLHmP","QM`BN`XQYd","RG@DTLHmP","QMHAIXFEVd","QMDARVAaH","RFPDXLHmP","RF@Q`vRbdLEC@","RF@QpvR@","QO@HyjAmd`","`II@B","`II@CFspqJp","`II@CF[@hM@prB`","`H@[T[|B`XN@PdM@p|@bHrBcDk@","RG@DXMj}F@","QM`BN`[L~b@","RG@DTMj}D@","QMHAIXFt~j@","QMDARVA}L@","RFPDXMj}D@","sKP@Di\\YZ@[TQXqaa`@","eF`BHD"]);},e.qu=function(t,A){var n,P;if(!A)return null;for(P=j(l.j7,H,6,(e.nu.length+31)/32|0,15,1),A=e.vu(A),e.cu(t.g,A),n=0;n<e.nu.length;n++)e._t(t.g,e.ou[n]),e.Pt(t.g,1,4)>0&&(P[n/32|0]|=1<<31-n%32);return P;},e.ru=function(t,A){var n,P;if(!A)return null;for(P=j(l.k7,e0,6,(e.nu.length+63)/64|0,14,1),A=e.vu(A),e.cu(t.g,A),n=0;n<e.nu.length;n++)e._t(t.g,e.ou[n]),e.Pt(t.g,1,4)>0&&(P[n/64|0]=rt(P[n/64|0],h1(1,63-n%64)));return P;},e.su=function(){var t,A;if(e.ou==null)for(A=new e._q(false),e.ou=j(l.j8,OA,26,e.nu.length,0,1),t=0;t<e.nu.length;t++)e.ou[t]=e.Pq(A,e.nu[t]),e.bw(e.ou[t],1);},e.tu=function(t){var A;if(t.f!=null){for(A=0;A<t.f.length;A++)if((t.c[A]&~t.f[A])!=0)return  false;}else return  false;return e.uu(t);},e.uu=function(t){return !t.d&&(t.d=e.Rq(new e._q(false),t.e)),!t.a&&(t.a=e.Rq(new e._q(false),t.b)),e.cu(t.g,t.d),e._t(t.g,t.a),e.Wt(t.g);},e.vu=function(t){var A,n;if(t.K){for(A=0;A<t.q;A++)if(pt(J(t.B[A],r0),0)){for(t=new e.vw(t),n=A;n<t.q;n++)pt(J(t.B[n],r0),0)&&(t.C[n]=-1);e.vl(t);}}return t;},e.wu=function(t,A,n){t.b=null,t.a=A,n==null?t.c=e.qu(t,A):t.c=n;},e.xu=function(t,A,n){t.e=null,t.d=A,n==null?t.f=e.qu(t,A):t.f=n;},e.yu=function(){e.pu(),this.g=new e.iu(),e.su();},e.zu=function(t){e.pu();var A,n,P,r,i,D;if(t==null)return null;for(A=j(l.f7,Y,6,t.length*8,15,1),P=0;P<t.length;P++)for(D=t[P],r=7;r>=0;r--)n=D&15,n>9&&(n+=7),A[P*8+r]=48+n<<24>>24,D>>=4;return T.ijb((i=A,S1.Mmb(),i));},e.Au=function(t){e.pu();var A,n,P,r;if(w.aAb(t).length==0||(w.aAb(t).length&7)!=0)return null;for(P=j(l.j7,H,6,w.aAb(t).length/8|0,15,1),n=0;n<w.aAb(t).length;n++)r=n/8|0,A=(w.Xzb(n,w.aAb(t).length),w.aAb(t).charCodeAt(n)-48),A>16&&(A-=7),P[r]<<=4,P[r]+=A;return P;},e.Bu=function(t,A){e.pu();var n,P,r,i;for(i=0,P=0,r=0,n=0;n<t.length;n++)i+=Pe(t[n]&A[n]),P+=Pe(t[n]),r+=Pe(A[n]);return i/d.Math.sqrt(P*r);},e.Cu=function(t,A){e.pu();var n,P,r;for(r=0,n=0,P=0;P<t.length;P++)r+=Pe(t[P]&A[P]),n+=Pe(t[P]|A[P]);return r/n;},N(180,1,{},e.yu),l.Z7=O(180),e.Eu=function(t){t.j=1,t.k=10,t.n=400,t.f=400,t.i=true,t.d="black",t.b=new y.Mj(),t.a=new y.Mj(),t.c=new T.Ujb(),t.e=new v1.Tfb(12);},e.Fu=function(t,A){var n,P;return n=(P=v1.Rfb(t.e,A),new j0.jgb(P)).b,n;},e.Gu=function(t){return t.k;},e.Hu=function(t){var A,n,P,r,i,D;if(i='<svg id="'+e.Ou(t.g!=null?t.g:"mol"+e.Du)+K1+'xmlns="http://www.w3.org/2000/svg" version="1.1" width="'+t.n+'px" height="'+t.f+'px" viewBox="0 0 '+t.n+" "+t.f+`">
`,D=t.i?"<style> #"+e.Ou(t.g!=null?t.g:"mol"+e.Du)+" { pointer-events:none; } #"+e.Ou(t.g!=null?t.g:"mol"+e.Du)+` .event  { pointer-events:all; } line { stroke-linecap:round; } polygon { stroke-linejoin:round; } </style>
`:'<g style="font-size:'+t.k+'px; fill-opacity:1; stroke-opacity:1; fill:black; stroke:black; font-weight:normal; text-rendering:optimizeLegibility; font-family:sans-serif; stroke-linejoin:round; stroke-linecap:round; stroke-dashoffset:0;">',i+="  ",i+=D,t.i){for(r=new y.Bqb(t.b);r.a<r.c.a.length;)P=y.Aqb(r),e.Iu(t,P);for(n=new y.Bqb(t.a);n.a<n.c.a.length;)A=y.Aqb(n),e.Iu(t,A);}return t.i||e.Iu(t,"</g>"),i+t.c.a+"</svg>";},e.Iu=function(t,A){T.Sjb(t.c,"  "),T.Sjb(t.c,A),T.Sjb(t.c,k1);},e.Ju=function(t,A,n){e._c(),e.de.call(this,t,A),e.Eu(this),this.g=n,++e.Du;},e.Ou=function(t){var A,n,P,r,i;for(i=new T.Ujb(),r=w.aAb(t).length,P=0;P<r;P++)switch(A=aD(t,P,w.aAb(t).length),n=A&n1,A){case 38:i.a+="&amp;";break;case 60:i.a+="&lt;";break;case 62:i.a+="&gt;";break;case 34:i.a+="&quot;";break;case 39:i.a+="&apos;";break;default:i.a+=String.fromCharCode(n);}return i.a;},N(259,121,{121:1},e.Ju),U.qb=function(t){var A;A='<line x1="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.a),(R.Dmb(),R.zmb))))+K1+'y1="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.c),R.zmb)))+K1+'x2="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.b),R.zmb)))+K1+'y2="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.d),R.zmb)))+K1+sr+this.d+K1+fP+this.j+DP,e.Iu(this,A);},U.rb=function(t){var A,n;A=T.Vhb(R.ukb(R.qkb(new R.vkb(3*this.j),(R.Dmb(),R.zmb)))),n='<line stroke-dasharray="'+A+","+A+K1+'x1="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.a),R.zmb)))+K1+'y1="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.c),R.zmb)))+K1+'x2="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.b),R.zmb)))+K1+'y2="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t.d),R.zmb)))+K1+sr+this.d+K1+"stroke-width:"+this.j+DP,e.Iu(this,n);},U.sb=function(t){var A,n;for(n=new T.Wjb('<polygon points="'),A=0;A<t.a;A++)T.Qjb(n,c1(d.Math.round(t.b[A]))),n.a+=",",T.Qjb(n,c1(d.Math.round(t.c[A]))),n.a+=" ";T.Sjb(n,'" fill="'+this.d+K1+sr+this.d+K1+fP+this.j+DP),e.Iu(this,n.a);},U.tb=function(t,A,n){var P,r,i,D;i=(P=(D=v1.Rfb(this.e,t),new j0.jgb(D)).b,P),r='<text x="'+T.Vhb(R.ukb(R.qkb(new R.vkb(A-i/2),(R.Dmb(),R.zmb))))+K1+'y="'+T.Vhb(R.ukb(R.qkb(new R.vkb(n+this.k/3),R.zmb)))+K1+'stroke="none" font-size="'+this.e.b+K1+'fill="'+e.Ou(this.d)+'">'+e.Ou(t)+"</text>",e.Iu(this,r);},U.ub=function(t,A,n){var P;P='<circle cx="'+T.Vhb(R.ukb(R.qkb(new R.vkb(t+n/2),(R.Dmb(),R.zmb))))+K1+'cy="'+T.Vhb(R.ukb(R.qkb(new R.vkb(A+n/2),R.zmb)))+K1+'r="'+T.Vhb(R.ukb(R.qkb(new R.vkb(n/2),R.zmb)))+K1+'fill="'+this.d+DP,e.Iu(this,P);},U.vb=function(t){return e.Fu(this,t);},U.wb=function(){return e.Gu(this);},U.yb=function(t,A,n,P){var r,i;r=e.Ll(this.L,t),i='<circle id="'+e.Ou(this.g!=null?this.g:"mol"+e.Du)+":Atom:"+t+K1+$8+(r==0?"":'data-atom-map-no="'+r+K1)+'cx="'+T.Vhb(R.ukb(R.qkb(new R.vkb(n),(R.Dmb(),R.zmb))))+K1+'cy="'+T.Vhb(R.ukb(R.qkb(new R.vkb(P),R.zmb)))+K1+'r="8'+K1+a8,y.sj(this.a,i);},U.zb=function(t,A,n,P,r){var i;i='<line id="'+e.Ou(this.g!=null?this.g:"mol"+e.Du)+":Bond:"+t+K1+$8+'x1="'+T.Vhb(R.ukb(R.qkb(new R.vkb(A),(R.Dmb(),R.zmb))))+K1+'y1="'+T.Vhb(R.ukb(R.qkb(new R.vkb(n),R.zmb)))+K1+'x2="'+T.Vhb(R.ukb(R.qkb(new R.vkb(P),R.zmb)))+K1+'y2="'+T.Vhb(R.ukb(R.qkb(new R.vkb(r),R.zmb)))+K1+fP+8+K1+a8,y.sj(this.b,i);},U.Ab=function(t){this.j=ee(c1(d.Math.round(100*d.Math.max(t,1))))/100;},U.Bb=function(t){this.d="rgb("+((t&W0)>>16)+","+((t&Y0)>>8)+","+(t&255)+")";},U.Cb=function(t){this.k!=t&&(this.k=t,this.e=new v1.Tfb(t));},U.Db=function(t,A){return this.n=Jt(c1(d.Math.round(t.b))),this.f=Jt(c1(d.Math.round(t.a))),e.$d(this,t,A);},U.pb=function(){return e.Hu(this);},U.f=0,U.i=false,U.j=0,U.k=0,U.n=0,e.Du=0,l._7=O(259),e.Zu=function(t,A){var n;if(t.c!=null){for(n=0;n<t.c.length;n++)if(A==t.c[n]&&t.d[n])return  true;}return  false;},e.$u=function(t,A,n,P,r){this.a=t,this.b=A,this.g=n,this.f=P,this.e=r;},N(184,1,{184:1},e.$u),U.a=0,U.b=0,U.e=false,U.f=false,U.g=0,l.d8=O(184),e._u=function(t,A){!t.b&&(t.b=new Lt.j4()),Lt.d4(t.b,T.Aib(A));},e.av=function(t,A,n,P){var r,i,D,u,o,Q;if(r=e.Uk(A,t.d),e.jn(A,r,t.e),e.rn(A,r,t.o,false),e.gn(A,r,t.a),pt(t.c,0)&&(pt(J(t.c,2),0)?(t.c=J(t.c,-3),A.u[r]|=n0):A.u[r]&=an,e.vn(A,r,t.c,true)),t.b){for(Q=j(l.j7,H,6,t.b.d.a.length,15,1),o=0;o<t.b.d.a.length;o++)Q[o]=Lt.f4(t.b,o).a;e.pn(A,r,Q),w.uzb(t.b.d.a,0);}else if(String.fromCharCode(n).toLowerCase().charCodeAt(0)==n&&T.khb(n)){if(t.d!=5&&t.d!=6&&t.d!=7&&t.d!=8&&t.d!=15&&t.d!=16&&t.d!=33&&t.d!=34)throw Ut(new T.cC("SmilesParser: atomicNo "+t.d+" must not be aromatic. Position:"+(P-1)));A.u[r]|=n0;}else A.u[r]&=an;if(t.f)for(u=new y.Bqb(t.f);u.a<u.c.a.length;)D=y.Aqb(u),t.r=true,e.Cn(D,0,0),e.On(A,true),e.dl(A,D,r);return t.g!=-1&&t.d!=1&&(i=j(l.f7,Y,6,1,15,1),i[0]=t.g<<24>>24,e.nn(A,r,i)),r;},e.bv=function(t,A){var n;for(n=0;A<t.length&&(t[A]!=93||n!=0);)t[A]==91?++n:t[A]==93&&--n,++A;if(A==t.length)throw Ut(new T.cC("SmilesParser: No matching closing bracket found."));return A+1;},e.cv=function(t,A,n){var P,r;if(A<3||(P=A-1,(t[P]==43||t[P]==45)&&--P,!AA(t[P]&n1))||(--P,t[P]!=68&&t[P]!=82&&t[P]!=88&&t[P]!=122))return  false;for(n[0]=0;P>0&&tu(t[P-1]);)--P,++n[0];for(r=A+1;tu(t[P]);){if(t.length<=r||t[P]!==t[r])return  false;++P,++r;}return  true;},e.dv=function(t,A,n,P,r,i){var D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;if(A[n-1]==36){if(t.r=true,t.q=new y.Mj(),n+=e.hv(t,A,n-1,t.q)-1,A[n++]!=93){if(!i)throw Ut(new T.cC("SmilesParser: A positive recursive SMARTS followed by another one or by atom query features is not supported. Position:"+(n-1)));(t.k&128)==0&&(n=e.bv(A,n));}return n;}if(A[n-1]==42)t.d=6,t.c=rt(t.c,1);else if(A[n-1]==63)t.d=0;else if(Q=A[n-1]==33,Q&&(t.r=true,t.c=rt(t.c,1),++n),A[n-1]==82&&r&&(AA(A[n]&n1)||t.j&&A[n]==123))t.d=6,t.c=rt(t.c,1),--n,Q&&--n;else {if(L=new e.jv(),!e.ev(t,A,n-1,P,L))throw Ut(new T.cC("SmilesParser: Unexpected character in atom definition:'"+String.fromCharCode(A[n-1]&n1)+ti+(n-1)));if(t.d=L.a,n+=L.b-1,(t.k&3)!=2&&(t.g=9),r&&(A[n]==44||Q)){for(b=L.d,h=L.c,p=n-L.b;p<P;){if(!e.ev(t,A,p,P,L)){if(!Q)throw Ut(new T.cC("SmilesParser: Unexpected character in atom list:'"+String.fromCharCode(A[p]&n1)+Ai+p));break;}if(L.a==1){if(!Q)throw Ut(new T.cC("SmilesParser: Hydrogen is not supported in positive atom lists:'"+T.hjb((y.Gqb(A,p,P),y.Lqb(A,j(l.f7,Y,6,P-p,15,1),p,P)))+Ai+p));}else e._u(t,L.a),b=b|L.d,h=h|L.c;if(p+=L.b,Q&&A[p]!=59&&A[p]!=38||!Q&&A[p]!=44||Q&&A[p+1]!=33)break;++p,A[p]==33&&++p;}t.b&&t.b.d.a.length>1&&(t.g=-1,h?b||(t.c=rt(t.c,4)):t.c=rt(t.c,2)),n=p;}}for(g=new e.nv(A),k=j(l.j7,H,6,1,15,1),M=true;M;){if(A[n]==64){++n,A[n]==64&&(t.i=true,++n),t.p=true;continue;}if(A[n]==58){for(++n;AA(A[n]&n1);)t.o=10*t.o+A[n]-48,++n;continue;}if(A[n]==91)throw Ut(new T.cC("SmilesParser: nested square brackets found. Position:"+n));if(A[n]==93){++n,M=false;continue;}if(t.e=e.gv(A,n,k),k[0]!=0){n+=k[0],t.e==0&&(t.c=rt(t.c,nn));continue;}if(o=A[n]==33,o&&++n,A[n]==72){++n,n+=e.kv(g,n,1),D=0,g.c<=0&&g.b>=0&&(D=rt(D,128)),g.c<=1&&g.b>=1&&(D=rt(D,256)),g.c<=2&&g.b>=2&&(D=rt(D,512)),g.c<=3&&g.b>=3&&(D=rt(D,i0)),o?(t.c=rt(t.c,D),t.g=-1):g.b==g.c?t.g=g.c:(t.c=rt(t.c,J(X0,tA(D))),t.g=-1);continue;}if(A[n]==68||A[n]==100){++n,n+=e.kv(g,n,1),D=0,g.c<=0&&g.b>=0&&(D=rt(D,n0)),g.c<=1&&g.b>=1&&(D=rt(D,sA)),g.c<=2&&g.b>=2&&(D=rt(D,yA)),g.c<=3&&g.b>=3&&(D=rt(D,hA)),g.c<=4&&g.b>=4&&(D=rt(D,v0)),et(D,0)!=0&&(o?t.c=rt(t.c,D):pt(J(t.c,aA),0)?t.c=J(t.c,tA(D)):(D=ne(D,aA),t.c=rt(t.c,D)));continue;}if(A[n]==122&&t.j){++n,n+=e.kv(g,n,4),D=0,g.c<=0&&g.b>=0&&(D=rt(D,Ce)),g.c<=1&&g.b>=1&&(D=rt(D,ei)),g.c<=2&&g.b>=2&&(D=rt(D,ni)),g.c<=3&&g.b>=3&&(D=rt(D,Pi)),g.c<=4&&g.b>=4&&(D=rt(D,fn)),et(D,0)!=0&&(o?t.c=rt(t.c,D):pt(J(t.c,Qe),0)?t.c=J(t.c,tA(D)):(D=ne(D,Qe),t.c=rt(t.c,D)));continue;}if(A[n]==88){if(++n,n+=e.kv(g,n,1),S=(e.Rk(),e.Mk)[t.d],S==null)continue;q=S[0],s=e.gv(A,n,k),k[0]!=0&&(e.qo(t.d)?q+=s:t.d==6?q-=d.Math.abs(s):q-=s),D=0,q-g.c<=0&&q-g.b>=0&&(D=rt(D,I0)),q-g.c<=1&&q-g.b>=1&&(D=rt(D,LA)),q-g.c<=2&&q-g.b>=2&&(D=rt(D,q1)),et(D,0)!=0&&(o?t.c=rt(t.c,D):pt(J(t.c,$A),0)?t.c=J(t.c,tA(D)):(D=ne(D,$A),t.c=rt(t.c,D)));continue;}if(A[n]==65||A[n]==97){t.c=rt(t.c,o^A[n]==65?4:2),++n;continue;}if(A[n]==82){++n,n+=e.kv(g,n,3),D=0,g.c<=0&&g.b>=0&&(D=rt(D,8)),g.c<=1&&g.b>=1&&(D=rt(D,16)),g.c<=2&&g.b>=2&&(D=rt(D,32)),g.c<=3&&g.b>=3&&(D=rt(D,64)),g.b>3&&e.Hv(t.n,(o?"!R":"R")+g.b),et(D,0)!=0&&(o?t.c=rt(t.c,D):pt(J(t.c,120),0)?t.c=J(t.c,tA(D)):(D=ne(D,120),t.c=rt(t.c,D)));continue;}if(A[n]==114){if(++n,n+=e.kv(g,n,1),g.a){o?t.c=rt(t.c,384):t.c=rt(t.c,8);continue;}B=g.c,g.b>g.c&&e.Hv(t.n,(o?"!r":"r")+("{"+g.c+"-"+g.b+"}")),!o&&B>=3&&B<=7?t.c=rt(t.c,l.Ueb(B<<22)):g.b>g.c||e.Hv(t.n,(o?"!r":"r")+B);continue;}if(A[n]==118){++n,n+=e.kv(g,n,1),q=g.c,g.b>g.c&&e.Hv(t.n,(o?"!v":"v")+("{"+g.c+"-"+g.b+"}")),!o&&q<=14?t.a=q:g.b>g.c||e.Hv(t.n,(o?"!v":"v")+q);continue;}if(A[n]==94){if(++n,u=A[n++]-48,u<1||u>3)throw Ut(new T.cC("SmilesParser: Unsupported hybridization. Position:"+n));c=u==1?q1:u==2?LA:I0,o||(c=J($A,tA(c))),t.c=rt(t.c,c);continue;}if(A[n]==36){if(!o)throw Ut(new T.cC("SmilesParser: non-negated recursive SMARTS relating to preceding atom are not supported yet. Position:"+n));!t.f&&(t.f=new y.Mj()),t.r=true,n+=e.hv(t,A,n,t.f);continue;}if(r&&(A[n]==59||A[n]==38)){t.r=true,++n;continue;}if(r&&A[n]==44&&e.cv(A,n,k)){t.r=true,n+=k[0]+1;continue;}if(r&&A[n]==44&&(t.k&128)!=0){t.r=true,n+=1;break;}throw A[n]==44?Ut(new T.cC("SmilesParser: alternative atom definitions not supported. (Tip: enumerate SMARTS): '"+String.fromCharCode(A[n]&n1)+hr+n)):Ut(new T.cC("SmilesParser: unexpected character inside brackets: '"+String.fromCharCode(A[n]&n1)+hr+n));}return n;},e.ev=function(t,A,n,P,r){var i,D;if(r.d=true,r.c=true,A[n]==35){for(++n,t.r=true,r.a=0,r.b=1;n<P&&AA(A[n]&n1);)r.a=10*r.a+A[n]-48,++r.b,++n;if(r.a==0||r.a>=(e.Rk(),e.Lk).length)throw Ut(new T.cC("SmilesParser: Atomic number out of range. position:"+(n-1)));return  true;}if(A[n]>=65&&A[n]<=90){if(r.b=A[n+1]>=97&&A[n+1]<=122?2:1,r.a=(e.Rk(),e.po(T.Hjb(w.Azb(A,n,(D=r.b,S1.Mmb(),D))),321)),r.b==2&&r.a==0&&(r.b=1,r.a=e.po(T.Hjb(w.Azb(A,n,(i=r.b,i))),321)),r.d=false,r.a==0)throw Ut(new T.cC("SmilesParser: Unknown atom label. position:"+(n-1)));return  true;}return A[n]==65&&A[n+1]==115||A[n]==83&&A[n+1]==101?(r.b=2,r.a=(e.Rk(),e.po(T.Hjb(w.Azb(A,n,(i=r.b,S1.Mmb(),i))),321)),r.c=false,true):A[n]==99||A[n]==110||A[n]==111||A[n]==112||A[n]==115?(r.b=1,r.a=(e.Rk(),e.po(T.Hjb(w.Azb(A,n,(i=r.b,S1.Mmb(),i))),321)),r.c=false,true):false;},e.fv=function(t,A,n,P,r){if(A[n-1]==42)t.d=6,t.c=rt(t.c,1);else if(A[n-1]==63)t.d=0;else if((A[n-1]==65||A[n-1]==97)&&r)t.d=6,t.c=rt(t.c,1),t.c=rt(t.c,A[n-1]==65?4:2),t.r=true;else switch(String.fromCharCode(A[n-1]).toUpperCase().charCodeAt(0)){case 66:n<P&&A[n]==114?(t.d=35,++n):t.d=5;break;case 67:n<P&&A[n]==108?(t.d=17,++n):t.d=6;break;case 70:t.d=9;break;case 73:t.d=53;break;case 78:t.d=7;break;case 79:t.d=8;break;case 80:t.d=15;break;case 83:t.d=16;}return n;},e.gv=function(t,A,n){var P,r;if(n[0]=0,t[A]==43||t[A]==45){for(r=t[A],P=1,++n[0];t[A+n[0]]==r;)++P,++n[0];return P==1&&AA(t[A+1]&n1)&&(P=t[A+1]-48,++n[0]),r==43?P:-P;}return 0;},e.hv=function(t,A,n,P){var r,i,D,u,o;if(A[n+1]!=40)throw Ut(new T.cC("SmilesParser: '$' for recursive SMARTS must be followed by '('. position:"+n));for(u=1,i=n+2;i<A.length&&u>0;)A[i]==40?++u:A[i]==41&&--u,++i;if(u>0)throw Ut(new T.cC("SmilesParser: Missing closing ')' for recursive SMARTS. '('-position:"+(n+1)));if(D=new e.uw(16,16),o=new e.Jv(t.k),e.Fv(o,t.n.f),e.xv(o,D,A,n+2,i-1,true,true),e.On(D,true),w.szb(P.a,D),A[n-1]==33)for(r=0;r<D.q;r++)e.vn(D,r,r0,true);return i-n;},e.iv=function(t,A){this.n=t,this.k=A,this.j=(A&16)==0,this.d=-1,this.e=0,this.o=0,this.a=-1,this.g=-1,this.c=0;},N(283,1,{},e.iv),U.a=0,U.c=0,U.d=0,U.e=0,U.g=0,U.i=false,U.j=false,U.k=0,U.o=0,U.p=false,U.r=false,l.c8=O(283),e.jv=function(){this.a=-1;},N(363,1,{},e.jv),U.a=0,U.b=0,U.c=false,U.d=false,l.a8=O(363),e.kv=function(t,A,n){var P,r,i,D,u;if(t.a=false,t.d=A,AA(t.e[A]&n1)){for(u=e.lv(t),t.c=t.b=u,P=A-1;P>1&&t2(t.e[P-1]);)--P;for(;t.e[t.d]==44;){for(D=true,i=A-P,r=0;r<i;r++)if(t.e[P+r]!==t.e[t.d+1+r]){D=false;break;}if(!D)break;t.d+=1+i,u=e.lv(t),t.c>u?t.c=u:t.b<u&&(t.b=u);}return t.d-A;}return t.e[A]==123&&AA(t.e[A+1]&n1)?(++t.d,t.c=e.lv(t),t.e[t.d++]!=45||!AA(t.e[t.d]&n1)||(t.b=e.lv(t),t.e[t.d++]!=125)?0:t.d-A):(t.c=1,t.b=n,t.a=true,0);},e.lv=function(t){var A;return A=t.e[t.d++]-48,AA(t.e[t.d]&n1)&&(A=10*A+(t.e[t.d++]-48)),A;},e.mv=function(t){return "{"+t.c+"-"+t.b+"}";},e.nv=function(t){this.e=t;},N(364,1,{},e.nv),U.pb=function(){return e.mv(this);},U.a=false,U.b=0,U.c=0,U.d=0,l.b8=O(364),e.pv=function(t,A,n,P,r,i){var D;return D=e.Wk(t.k,A,n,P),y.Bnb(t.s,A+"_"+n,v(F(l.j7,1),H,6,15,[r,i])),y.Bnb(t.s,n+"_"+A,v(F(l.j7,1),H,6,15,[r,i])),y.Arb(t.a,A),y.Arb(t.a,n),D;},e.qv=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c;for(Q=j(l.j7,H,6,t.k.f,15,1),u=j(l.j7,H,6,t.k.f,15,1),o=j(l.j7,H,6,t.k.f,15,1),L=j(l.j7,H,6,t.k.f,15,1),n=e.Zl(t.k,0,A),P=e.Zl(t.k,1,A),u[0]=n,u[1]=P,o[0]=-1,o[1]=A,Q[n]=1,Q[P]=2,L[n]=-1,L[P]=n,D=1,s=1;D<=s&&Q[u[D]]<15;){for(c=u[D],h=0;h<e.$o(t.k,c);h++)if(r=e.Zo(t.k,c,h),r!=L[c]){if(i=e._o(t.k,c,h),r==n){for(o[0]=i,b=0;b<=s;b++)t.g[o[h]]||(t.g[o[h]]=true,++t.c);return;}e.Km(t.k,r)&&Q[r]==0&&(++s,u[s]=r,o[s]=i,Q[r]=Q[c]+1,L[r]=c);}++D;}},e.rv=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b;for(e.bw(t.k,7),s=false,h=j(l.j7,H,6,2,15,1),b=j(l.j7,H,6,2,15,1),L=j(l.j7,H,6,2,15,1),P=0;P<t.k.g;P++)if(!e.jq(t.k,P)&&e.im(t.k,P)==2){for(D=0;D<2;D++){for(h[D]=-1,L[D]=-1,A=e.Zl(t.k,D,P),Q=0;Q<e.$o(t.k,A);Q++)r=e._o(t.k,A,Q),r!=P&&(h[D]==-1&&(e.im(t.k,r)==257||e.im(t.k,r)==129)?(h[D]=e.Zo(t.k,A,Q),b[D]=r):L[D]=e.Zo(t.k,A,Q));if(h[D]==-1)break;}if(h[0]!=-1&&h[1]!=-1){for(o=e.im(t.k,b[0])==e.im(t.k,b[1]),u=0;u<2;u++)h[u]==e.Zl(t.k,0,b[u])&&(o=!o);for(i=0;i<2;i++)L[i]!=-1&&L[i]<h[i]&&(o=!o);e.Jn(t.k,P,o?2:1,false),s=true;}}for(n=0;n<t.k.g;n++)(e.im(t.k,n)==257||e.im(t.k,n)==129)&&e.Mn(t.k,n,1);return s;},e.sv=function(t){return t==61?2:t==35?4:t==36?32:t==58?8:t==62?16:t==126?127:1;},e.tv=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;for(n=0;n<t.k.f;n++)if(e.Km(t.k,n))for(o=j(l.j7,H,6,t.k.f,15,1),u=j(l.j7,H,6,t.k.f,15,1),Q=j(l.j7,H,6,t.k.f,15,1),u[0]=n,o[n]=1,Q[n]=-1,D=0,L=0;D<=L;){for(r=(o[u[D]]&1)==1?1:2,s=0;s<e.$o(t.k,u[D]);s++)if(P=e._o(t.k,u[D],s),e.fm(t.k,P)==r&&A[P]&&(i=e.Zo(t.k,u[D],s),o[i]==0)){if(r==1&&e.Km(t.k,i)){for(h=u[D];h!=-1;)e.Mn(t.k,e.Xo(t.k,i,h),r==1?2:1),r=3-r,i=h,h=Q[h];return e.sn(t.k,n,false),e.sn(t.k,i,false),t.b-=2,true;}u[++L]=i,Q[i]=u[D],o[i]=o[u[D]]+1;}++D;}return  false;},e.uv=function(t){var A,n,P,r;for(A=0;A<t.k.f;A++)if(e.Tl(t.k,A)==7&&e.Bl(t.k,A)==0&&e.vp(t.k,A)>3&&e.Qo(t.k,A)>0){for(r=0;r<e.$o(t.k,A);r++)if(n=e.Zo(t.k,A,r),P=e._o(t.k,A,r),e.fm(t.k,P)>1&&e.Hm(t.k,n)){e.im(t.k,P)==4?e.Mn(t.k,P,2):e.Mn(t.k,P,1),e.jn(t.k,A,e.Bl(t.k,A)+1),e.jn(t.k,n,e.Bl(t.k,n)-1),e.gn(t.k,A,-1);break;}}},e.vv=function(t){return t.r?"Unresolved SMARTS features:"+t.r:"";},e.wv=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt;for(e.bw(t.k,1),t.g=j(l.Feb,ht,6,t.k.g,16,1),t.c=0,s=0;s<t.k.g;s++)e.im(t.k,s)==8&&(e.Mn(t.k,s,1),t.g[s]=true,++t.c);for(p=j(l.Feb,ht,6,t.k.f,16,1),Nt=new e.Dt(t.k,3),M=j(l.Feb,ht,6,Nt.j.a.length,16,1),xt=0;xt<Nt.j.a.length;xt++){for(qt=y.xj(Nt.j,xt),M[xt]=true,I=qt,tt=0,it=I.length;tt<it;++tt)if(S=I[tt],!e.Km(t.k,S)){M[xt]=false;break;}if(M[xt]){for(Z=qt,ot=0,Qt=Z.length;ot<Qt;++ot)S=Z[ot],p[S]=true;for(Zt=y.xj(Nt.k,xt),V=Zt,z=0,G=V.length;z<G;++z)S=V[z],t.g[S]||(t.g[S]=true,++t.c);}}for(h=0;h<t.k.g;h++)!t.g[h]&&(Nt.c[h]&n1)!=0&&e.Km(t.k,e.Zl(t.k,0,h))&&e.Km(t.k,e.Zl(t.k,1,h))&&e.qv(t,h);for(b=0;b<t.k.g;b++)t.g[b]||(i=e.Zl(t.k,0,b),D=e.Zl(t.k,1,b),!p[i]&&!p[D]&&e.Km(t.k,i)&&e.Km(t.k,D)&&(t.g[b]=true,++t.c));for(e.bw(t.k,7),(t.q==2||t.q==1&&t.p)&&e.Dv(t),k=j(l.Feb,ht,6,t.k.g,16,1),t.k.g>=0&&T.$jb(t.g,0,k,0,t.k.g),Et=0;Et<Nt.j.a.length;Et++)if(M[Et]){for(qt=y.xj(Nt.j,Et),Dt=qt,lt=0,ct=Dt.length;lt<ct;++lt)if(ut=Dt[lt],!e.Ev(t,ut))for(e.Km(t.k,ut)&&(e.sn(t.k,ut,false),--t.b),S=0;S<e.$o(t.k,ut);S++)g=e._o(t.k,ut,S),t.g[g]&&(t.g[g]=false,--t.c);}for(e.Cv(t),bt=0;bt<Nt.j.a.length;bt++)if(M[bt]&&y.xj(Nt.k,bt).length==6){for(Zt=y.xj(Nt.k,bt),q=true,o=Zt,Q=0,L=o.length;Q<L;++Q)if(u=o[Q],!t.g[u]){q=false;break;}q&&(e.Bv(t,Zt[0]),e.Bv(t,Zt[2]),e.Bv(t,Zt[4]),e.Cv(t));}for(wt=5;wt>=4;wt--)do for(yt=false,u=0;u<t.k.g;u++)if(t.g[u]){for(P=0,B=0;B<2;B++)for(c=e.Zl(t.k,B,u),S=0;S<e.$o(t.k,c);S++)t.g[e._o(t.k,c,S)]&&++P;if(P==wt){e.Bv(t,u),e.Cv(t),yt=true;break;}}while(yt);for(;t.b>=2&&e.tv(t,k););if(A){if(t.b!=0)for(r=0;r<t.k.f;r++)e.Km(t.k,r)&&(e.sn(t.k,r,false),e.vn(t.k,r,2,true),--t.b,n=true);if(t.c!=0)for(u=0;u<t.k.g;u++)t.g[u]&&(t.g[u]=false,e.Mn(t.k,u,8),--t.c,n=true);}else for(r=0;r<t.k.f;r++)e.Km(t.k,r)&&e.op(t.k,r)!=0&&(e.sn(t.k,r,false),e.wn(t.k,r,32),--t.b);if(t.q==0||t.q==1&&!n){if(t.b!=0)throw Ut(new T.cC(ri));if(t.c!=0)throw Ut(new T.cC(ri));}},e.xv=function(t,A,n,P,r,i,D){var u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt,b1,u1,Ht,gt,w1,x1,L1,O1,U1,l1;for(t.k=A,e._v(t.k),t.s=new y.ztb(),t.a=new y.Brb(),t.r&&T.Ggb(t.r,0),t.b=0,t.p=false,Q=t.q!=0,It=null,c=j(l.j7,H,6,32,15,1),c[0]=-1,Bt=j(l.j7,H,6,16,15,1),u1=j(l.j7,H,6,16,15,1),b1=j(l.j7,H,6,16,15,1),jt=j(l.j7,H,6,16,15,1),ut=0;ut<16;ut++)Bt[ut]=-1;for(h=0,z=-1,Ht=false,yt=false,G=false,p=0,k=1,B=0;n[P]<=32;)++P;for(;P<r;){if(gt=n[P++]&n1,T.khb(gt)||gt==42||gt==63||gt==33&&Q&&Ht||gt==35&&Q&&Ht||gt==36&&Q&&Ht){if(b=new e.iv(t,t.j|t.q),Ht){if((t.j&128)!=0){if(V=new e.Lv(P-1),P=e.dv(b,n,P,r,true,true),n[P-1]!=93){for(;n[P-1]!=93;)P=e.dv(b,n,P+1,r,true,true),++V.a;t.f.add(V);}}else P=e.dv(b,n,P,r,Q,false);}else P=e.fv(b,n,P,r,Q);if(Ht=false,b.q&&y.xj(b.q,0)){z=c[p],c[p]=A.q,e.Yk(A,b.q?y.xj(b.q,0):null),z!=-1&&k!=512&&(g=e.Wk(t.k,z,z,k),B!=0&&(t.p=true,e.Ln(t.k,g,B,true),e.Sk(t.k,g))),k=1,B=0;continue;}if(b.d==-1&&gt!=63)throw Ut(new T.cC("SmilesParser: unknown element label found. Position:"+(P-1)));b.r&&(t.p=true),s=e.av(b,t.k,gt,P),e.Km(t.k,s)&&++t.b,z=c[p],z!=-1&&k!=512&&(g=e.Wk(t.k,z,s,k),B!=0&&(t.p=true,e.Ln(t.k,g,B,true),e.Sk(t.k,g))),k=1,B=0,c[p]=s,h!=0&&(e.tn(t.k,s,h),h=0),D&&(Zt=It?y.opb(It,T.Aib(z)):null,Zt&&e.Nv(Zt,s,(wt=P,b.d==1&&e.Ml(t.k,s)==0,wt)),b.p&&(!It&&(It=new y.fwb()),it=b.g==9?0:b.g,y.Zvb(It,T.Aib(s),new e.Tv(t,s,P-2,z,it,b.i))));continue;}if(gt==46){c[p]=-1,k=512;continue;}if(gt==45||gt==61||gt==35||gt==36||gt==58||gt==47||gt==92||gt==60||gt==126||gt==33||gt==64){if(Ht)throw Ut(new T.cC("SmilesParser: unexpected bond symbol inside square brackets: '"+String.fromCharCode(gt)+hr+(P-1)));for(I=0;gt==45||gt==61||gt==35||gt==36||gt==58||gt==47||gt==92||gt==60||gt==126||gt==33||gt==64;){if(gt==33){if(gt=n[P++]&n1,gt==64)B|=128;else if(gt==45&&n[P]==62||gt==60&&n[P]==45)I|=16,++P;else if(gt==45)I|=1;else if(gt==61)I|=2;else if(gt==35)I|=4;else if(gt==36)I|=32;else if(gt==58)I|=8;else throw Ut(new T.cC("SmilesParser: bond symbol '"+String.fromCharCode(gt)+"' not allowed after '!'. Position:"+(P-1)));}else if(gt==64?B|=256:gt==61?k=2:gt==35?k=4:gt==36?k=32:gt==58?k=8:gt==126?B|=15:gt==47?D&&(k=257):gt==92?D&&(k=129):(gt==45&&n[P]==62||gt==60&&n[P]==45)&&(k=16,++P),n[P]==44)for(B|=(k==16?62:gt)==61?2:(k==16?62:gt)==35?4:(k==16?62:gt)==36?32:(k==16?62:gt)==58?8:(k==16?62:gt)==62?16:(k==16?62:gt)==126?127:1;n[P]==44;)n[P+1]==60&&n[P+2]==45||n[P+1]==45&&n[P+2]==62?(B|=16,P+=3):(B|=e.sv(n[P+1]&n1),P+=2);if(n[P]==59){++P,gt=n[P++]&n1;continue;}I!=0&&(B|=127&~I);break;}continue;}if(gt<=32){P=r;continue;}if(AA(gt)){if(xt=gt-48,Ht){for(;P<r&&AA(n[P]&n1);)xt=10*xt+n[P]-48,++P;h=xt;}else {if(M=yt?P-3:P-2,ot=n[M]==45||n[M]==47||n[M]==92||n[M]==61||n[M]==35||n[M]==36||n[M]==58||n[M]==62||n[M]==126,yt&&P<r&&AA(n[P]&n1)&&(xt=10*xt+n[P]-48,yt=false,++P),xt>=Bt.length){if(xt>=100)throw Ut(new T.cC("SmilesParser: ringClosureAtom number out of range: "+xt));for(Et=Bt.length,bt=Bt.length;bt<=xt;)bt=d.Math.min(100,bt+16);for(Bt=(w.Mzb(bt),y.Lqb(Bt,j(l.j7,H,6,bt,15,1),0,bt)),u1=(w.Mzb(bt),y.Lqb(u1,j(l.j7,H,6,bt,15,1),0,bt)),b1=(w.Mzb(bt),y.Lqb(b1,j(l.j7,H,6,bt,15,1),0,bt)),jt=(w.Mzb(bt),y.Lqb(jt,j(l.j7,H,6,bt,15,1),0,bt)),Dt=Et;Dt<bt;Dt++)Bt[Dt]=-1;}if(Bt[xt]==-1)Bt[xt]=c[p],u1[xt]=P-1,b1[xt]=ot?k:-1,jt[xt]=ot?B:0;else {if(Bt[xt]==c[p])throw Ut(new T.cC("SmilesParser: ring closure to same atom"));D&&It&&(Zt=y.opb(It,T.Aib(Bt[xt])),Zt&&e.Nv(Zt,c[p],u1[xt]),Zt=y.opb(It,T.Aib(c[p])),Zt&&e.Nv(Zt,Bt[xt],P-1)),b1[xt]!=-1?k=b1[xt]:k==257?k=129:k==129&&(k=257),u=Bt[xt],o=c[p],g=e.pv(t,u,o,k,u1[xt],P-1),jt[xt]!=0&&(B=jt[xt]),B!=0&&(t.p=true,e.Ln(t.k,g,jt[xt],true),e.Sk(t.k,g)),Bt[xt]=-1;}k=1,B=0;}continue;}if(gt==43)throw Ut(new T.cC("SmilesParser: '+' found outside brackets. Position:"+(P-1)));if(gt==40){if(c[p]==-1){G=true;continue;}++p,c.length==p&&(c=y.Jqb(c,c.length+32)),c[p]=c[p-1];continue;}if(gt==41){if(p==0){if(!G)throw Ut(new T.cC("SmilesParser: Closing ')' without opening counterpart. Position:"+(P-1)));c[0]=-1,G=false;continue;}--p;continue;}if(gt==91){Ht=true;continue;}if(gt==93)throw Ut(new T.cC("SmilesParser: closing bracket at unexpected position:"+(P-1)));if(gt==37){yt=true;continue;}throw Ut(new T.cC("SmilesParser: unexpected character outside brackets: '"+String.fromCharCode(gt)+ti+(P-1)));}if(k!=1)throw Ut(new T.cC("SmilesParser: dangling open bond; position:"+P));for(lt=0;lt<Bt.length;lt++)if(Bt[lt]!=-1)throw Ut(new T.cC("SmilesParser: dangling ring closure: "+lt+"; position:"+P));for(tt=e.lp(t.k),e.Qn(t.k,true),e.bw(t.k,1),L=0;L<t.k.q;L++)if(e.El(t.k,L)!=null){if(Z=e.Fl(t.k,L)[0],t.p||t.q==2){if(Z!=9)if(t.i)for(Qt=0;Qt<Z;Qt++)e.Wk(t.k,L,e.Uk(t.k,1),1);else Z==0&&e.vn(t.k,L,1792,true),Z==1&&e.vn(t.k,L,1664,true),Z==2&&e.vn(t.k,L,1408,true),Z==3&&e.vn(t.k,L,896,true);}else {if(Z==9&&(Z=0),!e.Lm(t.k,L)&&(!e.Km(t.k,L)||e.Tl(t.k,L)==6&&e.Bl(t.k,L)==0)){for(l1=e.mo(e.Tl(t.k,L)),q=false,w1=e.vp(t.k,L),w1-=e.mm(t.k,L,w1),w1+=Z,e.Km(t.k,L)&&++w1,L1=l1,O1=0,U1=L1.length;O1<U1;++O1)if(x1=L1[O1],w1<=x1){q=true,x1==w1+2?e.wn(t.k,L,48):x1==w1+1?e.wn(t.k,L,32):(x1!=w1||x1!=l1[0])&&e.gn(t.k,L,w1);break;}q||e.gn(t.k,L,w1);}if(t.i||!e.Aq(t.k,L))for(Qt=0;Qt<Z;Qt++)e.Wk(t.k,L,e.Uk(t.k,1),1);}}else !t.i&&(t.p||t.q==2)&&(Z=e.cp(t.k,L),Z>=1&&e.vn(t.k,L,128,true),Z>=2&&e.vn(t.k,L,256,true),Z>=3&&e.vn(t.k,L,512,true),Z>=4&&e.vn(t.k,L,i0,true));if(!t.i&&(t.p||t.q==2)&&e.qq(t.k,false),e.bw(t.k,1),e.uv(t),e.wv(t,Q,t.p),t.k.t=null,e.Qn(t.k,false),D&&(e.rv(t),It)){for(Nt=(qt=new y.Oob(It).a.bd().Tb(),new y.Vob(qt));Nt.a.dd();)Zt=(S=Nt.a.ed(),S.hd()),e.un(t.k,tt[Zt.a],e.Ov(Zt,tt),false);e.vq(t.k,0);}e.vq(t.k,0),i&&(ct=new X.CA(t.d),pt(t.n,0)&&X.yA(ct,t.n),X.jA(ct,t.k),D&&e.ow(t.k)),(t.p||t.q==2)&&(e.On(t.k,true),e.Cq(t.k),e.Dq(t.k));},e.yv=function(t,A,n,P,r){e.xv(t,A,n,0,n.length,P,r);},e.zv=function(t,A){var n;return A==null?null:e.Av(t,w.Czb((n=A,S1.Mmb(),n)));},e.Av=function(t,A){var n,P,r,i,D,u,o,Q,L,s;for(r=Lt.I3(A,0);r>0&&A[r-1]==45;)r=Lt.I3(A,r+1);for(i=r==-1?-1:Lt.I3(A,r+1);i>0&&A[i-1]==45;)i=Lt.I3(A,i+1);if(i==-1)throw Ut(new T.cC("Missing one or both separators ('>')."));if(Lt.I3(A,i+1)!=-1)throw Ut(new T.cC("Found more than 2 separators ('>')."));for(s=new At.zH(),L=0,P=0;P<A.length;){for(;P<A.length&&A[P]==46;)++P;if(P==A.length)break;for(Q=P,o=-1,D=0,n=P;n<A.length;n++){if(A[n]==40){++D;continue;}if(A[n]==41){if(--D<0)throw Ut(new T.cC("Found closing bracket without matching opening one."));continue;}if(D==0&&(A[n]==46&&(t.o||n+1<A.length&&A[n+1]==46)||A[n]==62&&(n==0||A[n-1]!=45))){o=n;break;}}if(o==-1&&(o=A.length),Q<o){if(u=new e.tw(),e.xv(t,u,A,Q,o,true,true),t.q==1&&t.p)return e.Av(new e.Jv(t.j|2),A);L==0?(y.sj(s.g,u),s.d=-1):L==1?y.sj(s.a,u):(y.sj(s.f,u),s.d=-1);}for(P=o;P<A.length&&A[P]==62;)++P,++L;}return s;},e.Bv=function(t,A){var n,P,r,i;for(e.im(t.k,A)==1&&e.Mn(t.k,A,2),r=0;r<2;r++)for(n=e.Zl(t.k,r,A),e.Km(t.k,n)&&(e.sn(t.k,n,false),--t.b),i=0;i<e.$o(t.k,n);i++)P=e._o(t.k,n,i),t.g[P]&&(t.g[P]=false,--t.c);},e.Cv=function(t){var A,n,P,r,i,D,u;do for(u=false,n=0;n<t.k.g;n++)if(t.g[n]){for(i=false,r=0;r<2;r++){for(A=false,P=e.Zl(t.k,r,n),D=0;D<e.$o(t.k,P);D++)if(n!=e._o(t.k,P,D)&&t.g[e._o(t.k,P,D)]){A=true;break;}if(!A){i=true;break;}}i&&(u=true,e.Bv(t,n));}while(u);},e.Dv=function(t){var A,n,P,r,i,D;for(A=0;A<t.k.g;A++)if(t.g[A]){for(i=true,r=0;r<2&&i;r++)for(n=e.Zl(t.k,r,A),D=0;D<e.$o(t.k,n)&&i;D++)A!=e._o(t.k,n,D)&&t.g[e._o(t.k,n,D)]&&(i=false);if(i)for(e.Mn(t.k,A,8),--t.c,P=0;P<2;P++)n=e.Zl(t.k,P,A),e.Km(t.k,n)&&(e.sn(t.k,n,false),--t.b);}},e.Ev=function(t,A){var n,P;return !e.Ft(e.Tl(t.k,A))||e.Tl(t.k,A)==6&&(!e.Km(t.k,A)||e.Bl(t.k,A)>0)||(n=e.El(t.k,A)==null||e.Fl(t.k,A)[0]==9?0:e.Fl(t.k,A)[0],P=e.jp(t.k,A)-n,P<1)?false:e.Tl(t.k,A)==16||e.Tl(t.k,A)==34||e.Tl(t.k,A)==52?e.$o(t.k,A)==2&&e.Bl(t.k,A)<=0?false:P!=2:true;},e.Fv=function(t,A){t.f=A;},e.Gv=function(t,A){t.n=A;},e.Hv=function(t,A){t.e&&(!t.r&&(t.r=new T.Ujb()),T.Sjb(t.r," "),T.Sjb(t.r,A));},e.Iv=function(){e.Jv.call(this,0);},e.Jv=function(t){this.j=t&-4,this.q=t&3,this.o=(t&32)!=0,this.e=(t&64)!=0,this.i=(t&8)!=0,this.d=2,(t&4)!=0&&(this.d|=1),this.i&&(this.d&=-3);},N(113,1,{},e.Iv,e.Jv),U.b=0,U.c=0,U.d=0,U.e=false,U.i=false,U.j=0,U.n=0,U.o=false,U.p=false,U.q=0,l.h8=O(113),e.Kv=function(t,A){return r8(A.b,t.b);},e.Lv=function(t){this.b=t,this.a=1;},N(216,1,{216:1,32:1},e.Lv),U.Sb=function(t){return e.Kv(this,t);},U.a=0,U.b=0,l.e8=O(216),e.Nv=function(t,A,n){if(!t.c){if(t.e.a.length==4){t.c=true;return;}y.sj(t.e,new e.Uv(t,A,n));}},e.Ov=function(t,A){var n,P,r;if(t.c)return 3;for(r=new y.Bqb(t.e);r.a<r.c.a.length;)P=y.Aqb(r),P.a!=Un&&P.a!=St&&(P.a=A[P.a]);switch(n=false,t.e.a.length){case 2:if(y.xj(t.e,0).a>=Un||y.xj(t.e,1).a>=Un)return 3;n=e.Rv(t,A);break;case 3:y.sj(t.e,new e.Uv(t,St,t.b));case 4:n=e.Sv(t);break;default:return 3;}return t.d^n?1:2;},e.Pv=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it;if(p=t.f.k,r=y.xj(t.e,n),it=r.a,s=A[t.a],B=e.op(p,it)>0&&p.j[it]==p.e[it],g=y.xrb(t.f.a,t.f.t[it]),!B&&!g)return D=p.i[it][2],p.C[D]==1&&(D=t.f.t[D],(n==0?D>it:t.f.t[p.i[it][1]]>D)&&(P=!P)),P;if(B&&(i=p.i[it][n],D=t.f.t[i],D>it&&(P=!P,p.C[i]==1&&(P=!P))),g){for(q=p.e[it],b=j(l.j7,H,6,q-1,15,1),c=j(l.j7,H,6,q-1,15,1),S=0,k=0,I=0;k<q;k++)D=p.i[it][k],D!=s&&(b[I]=D,Q=e.Xo(p,it,D),tt=y.ynb(t.f.s,t.f.t[it]+"_"+t.f.t[D]),c[I++]=tt==null?-1:Q,tt!=null&&++S);if(B)b[0]<it&&(P=!P);else switch(S){case 1:if(e.op(p,it)>0)break;M=c[0]>=0,h=b[M?0:1],V=b[M?1:0],V>it?h>it&&p.C[V]!=1&&(P=!P):h<it&&(P=!P);break;case 2:u=e.Qv(t,c[0],it),o=e.Qv(t,c[1],it),ot=y.ynb(t.f.s,it+"_"+t.f.t[u]),G=y.ynb(t.f.s,it+"_"+t.f.t[o]),L=t.b,n==0?(Z=ot[1]<L?ot[1]:ot[0],z=G[1]<L?G[1]:G[0]):(Z=ot[0]<L?ot[1]:ot[0],z=G[0]<L?G[1]:G[0]),Z<z!=u<o&&(P=!P);}}return P;},e.Qv=function(t,A,n){return t.f.k.D[0][A]==n?t.f.k.D[1][A]:t.f.k.D[0][A];},e.Rv=function(t,A){var n,P;if(P=false,t.f.t==null)for(t.f.t=j(l.j7,H,6,A.length,15,1),n=t.f.t.length;--n>=0;)t.f.t[A[n]]=n;return P=e.Pv(t,A,0,P),P=e.Pv(t,A,1,P),P;},e.Sv=function(t){var A,n,P;for(n=false,A=1;A<t.e.a.length;A++)for(P=0;P<A;P++)y.xj(t.e,P).a>y.xj(t.e,A).a&&(n=!n),y.xj(t.e,P).b>y.xj(t.e,A).b&&(n=!n);return n;},e.Tv=function(t,A,n,P,r,i){this.f=t,r!=0&&r!=1?this.c=true:(this.a=A,this.b=n,this.d=i,this.e=new y.Mj(),P!=-1&&e.Nv(this,P,n-1),P!=-1&&r==1&&e.Nv(this,Un,n+1));},N(217,1,{217:1},e.Tv),U.a=0,U.b=0,U.c=false,U.d=false,l.g8=O(217),e.Uv=function(t,A,n){this.c=t,this.a=A,this.b=n;},N(182,1,{182:1},e.Uv),U.pb=function(){return "["+(this.a==Un?"h":this.a==St?"lp":e.Il(this.c.f.k,this.a))+this.b+"]";},U.a=0,U.b=0,l.f8=O(182),e.Wv=function(t){t.c=new y.Mj();},e.Xv=function(t,A){var n,P,r,i;if(i=t.c.a.length,i==0)return y.rj(t.c,0,A),0;for(r=1;2*r<=i;)r<<=1;for(P=r,--r;P!=0;){if(P>>=1,r>=i){r-=P;continue;}if(n=T.ejb(A,y.xj(t.c,r)),n==0)return  -1;if(P==0)break;n<0?r-=P:r+=P;}return r<i&&T.ejb(A,y.xj(t.c,r))>0&&++r,y.rj(t.c,r,A),r;},e.Yv=function(t,A){var n,P,r,i;if(i=t.c.a.length,i==0)return  -1;for(r=1;2*r<=i;)r<<=1;for(P=r,--r;P!=0;){if(P>>=1,r>=i){r-=P;continue;}if(n=T.ejb(A,y.xj(t.c,r)),n==0)return r;if(P==0)break;n<0?r-=P:r+=P;}return  -1;},e.Zv=function(){e.Wv(this);},N(102,1,ii,e.Zv),U.bc=function(t){return e.Yv(this,t);},l.i8=O(102),e._v=function(t){e.jl(t),t.d=null;},e.aw=function(t,A){e.pl(t,A),t.d&&(A.T=0);},e.bw=function(t,A){var n,P,r,i;if(e.Eo(t,A),(A&~t.T)!=0){for((t.T&15)!=0&&t.q>1&&e.ei(t.J[0],t.J[1])&&X.jA(new X.CA(0),t),t.c&&(A|=128),n=0;n<t.q;n++)t.u[n]&=-67223560;for(P=0;P<t.g;P++)t.F[P]&=-64;r=0,i=0,(A&32)!=0?(r=32,i=1):(A&64)!=0&&(r=64,i=3),(A&128)!=0&&(r|=128,i|=32),t.d=new e.Og(t,i),e.Jg(t.d),e.Kg(t.d),e.Ig(t.d),e.sw(t)&&(t.d=new e.Og(t,i)),t.T|=24|r;}},e.cw=function(t,A){return e.wg(t.d,A);},e.dw=function(t,A){return e.lg(t.d,A);},e.ew=function(t){var A;switch(e.bw(t,31),A=t.I&n1,t.I&fA){case q1:return null;case n0:return A==1?"meso":""+A+" meso diastereomers";case 0:return "unknown chirality";case Yn:return "both enantiomers";case sA:return "this enantiomer";case 327680:return "this or other enantiomer";case te:return "two epimers";default:return A==1?"one stereo isomer":""+A+" stereo isomers";}},e.fw=function(t){var A;return A=new e.uw(t.q,t.r),e.ol(t,A),A;},e.gw=function(t){var A,n,P,r,i,D,u;for(e.bw(t,15),D=j(l.j7,H,6,3,15,1),n=0;n<t.f;n++)(t.u[n]&S0)!=0&&(u=(t.u[n]&Q1)>>18,u!=0&&(D[u]=d.Math.max(D[u],(t.u[n]&Q1)>>18!=1&&(t.u[n]&Q1)>>18!=2?-1:(t.u[n]&F0)>>20)));for(r=0;r<t.g;r++)((t.F[r]&3)==1||(t.F[r]&3)==2)&&t.H[r]==1&&(u=(t.F[r]&768)>>8,u!=0&&(D[u]=d.Math.max(D[u],(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10)));for(i=j(l.j7,E1,7,3,0,2),i[1]=j(l.j7,H,6,1+D[1],15,1),i[2]=j(l.j7,H,6,1+D[2],15,1),A=0;A<t.f;A++)(t.u[A]&S0)!=0&&(u=(t.u[A]&Q1)>>18,u!=0&&++i[u][(t.u[A]&Q1)>>18!=1&&(t.u[A]&Q1)>>18!=2?-1:(t.u[A]&F0)>>20]);for(P=0;P<t.g;P++)((t.F[P]&3)==1||(t.F[P]&3)==2)&&t.H[P]==1&&(u=(t.F[P]&768)>>8,u!=0&&++i[u][(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10]);return i;},e.hw=function(t){var A,n;return A=j(l.j7,H,6,t.q,15,1),n=e.hp(t,A,false,false),e.iw(t,A,n);},e.iw=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;for(B=j(l.j8,OA,26,n,0,1),D=j(l.j7,H,6,n,15,1),Q=j(l.j7,H,6,n,15,1),i=j(l.j7,H,6,t.q,15,1),r=0;r<t.q;r++)A[r]!=-1&&(i[r]=D[A[r]]++);for(o=0;o<t.r;o++)c=A[t.D[0][o]],g=A[t.D[1][o]],c==g&&c!=-1&&++Q[c];for(k=0;k<n;k++)B[k]=new e.uw(D[k],Q[k]),e.aw(t,B[k]);for(P=0;P<t.q;P++)A[P]!=-1&&e.ll(t,B[A[P]],P,0,0);for(u=0;u<t.r;u++)c=A[t.D[0][u]],g=A[t.D[1][u]],c==g&&c!=-1&&e.nl(t,B[c],u,0,0,i,false);for(s=B,h=0,b=s.length;h<b;++h)L=s[h],e.cn(L,1),e.cn(L,2);return B;},e.jw=function(t){return e.bw(t,15),!t.d&&(t.f<2||!e.ei(t.J[0],t.J[1]))&&(t.d=new e.Ng(t)),t.d?e.rg(t.d):null;},e.kw=function(t){return e.bw(t,15),!t.d&&(t.f<2||!e.ei(t.J[0],t.J[1]))&&(t.d=new e.Ng(t)),t.d?e.mg(t.d):null;},e.lw=function(t){var A,n;for(e.bw(t,31),n=0,A=0;A<t.f;A++)(t.u[A]&3)!=0&&(t.u[A]&4)==0&&++n;return n;},e.mw=function(t,A){return e.vg(t.d,A);},e.nw=function(t,A){t.c=A,t.T&=-144;},e.ow=function(t){e.bw(t,31),t.d&&e.Lg(t.d);},e.pw=function(t){var A,n;for(e.bw(t,15),t.L=false,A=0;A<t.q;A++)t.u[A]&=AP,(t.u[A]&3)!=0?t.u[A]|=ln:t.u[A]&=tP;for(n=0;n<t.r;n++)(t.F[n]&3)!=0&&e.fm(t,n)==2?t.H[n]=386:t.H[n]&=-385;t.T&=-249;},e.qw=function(t,A,n){var P,r,i,D,u,o,Q,L,s;if(s=t.u[A]&3,s==1||s==2){if(u=false,t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2&&t.C[A]<=7)for(D=0;D<t.j[A];D++){for(P=t.i[A][D],L=0,Q=j(l.j7,H,6,3,15,1),o=0;o<t.j[P];o++)Q[L]=t.i[P][o],Q[L]!=A&&++L;L==2&&Q[0]<Q[1]^n[Q[0]]<n[Q[1]]&&(u=!u);}else for(D=1;D<t.j[A];D++)for(o=0;o<D;o++)r=t.i[A][D],i=t.i[A][o],r<i^n[r]<n[i]&&(u=!u);u&&(s=s==1?2:1);}return s;},e.rw=function(t){var A,n,P,r,i,D;for(e.Bq(t),e.bw(t,31),P=0;P<t.f;P++){if(((t.u[P]&Q1)>>18==1||(t.u[P]&Q1)>>18==2)&&((t.u[P]&S0)==0||(t.u[P]&3)==3))throw Ut(new T.cC(Tr));if((t.u[P]&q1)!=0)throw Ut(new T.cC(br));if(((t.u[P]&3)==1||(t.u[P]&3)==2)&&t.o[P]==0){for(A=j(l.h7,Ft,6,t.j[P],15,1),i=0;i<t.j[P];i++)A[i]=e.Yl(t,P,t.i[P][i]);for(r=1;r<t.j[P];r++)if(!e.Sm(t,t.k[P][r],P)){for(D=0;D<r;D++)if(!e.Sm(t,t.k[P][D],P)&&(n=d.Math.abs(e.oo(A[r],A[D])),n<.08||n>J8))throw Ut(new T.cC(cr));}}}},e.sw=function(t){var A,n,P,r,i,D,u;for(u=false,n=0;n<t.f;n++)((t.u[n]&S0)==0||(t.u[n]&3)==3)&&(t.u[n]&=AP);for(i=0;i<t.g;i++)(e.fm(t,i)!=1||(t.F[i]&3)==0||(t.F[i]&3)==3)&&(t.F[i]&=-32513);if(t.L){if((t.I&fA)!=n0){for(D=j(l.Feb,ht,6,t.f,16,1),P=0;P<t.f;P++)(t.u[P]&S0)!=0&&(t.u[P]&3)!=3&&(t.u[P]&Q1)>>18==1&&(D[P]=true);for(r=0;r<t.f;r++)(t.u[r]&S0)!=0&&(t.u[r]&3)!=3&&(e.on(t,r,1,0),u=true);for(A=0;A<t.f;A++)D[A]&&(e.un(t,A,1,false),e.on(t,A,1,-1),u=true);}t.L=false;}return e.cn(t,1),e.cn(t,2),u;},e.tw=function(){e.Rk(),e.Fq.call(this);},e.uw=function(t,A){e.Rk(),e.Gq.call(this,t,A);},e.vw=function(t){e.Rk(),e.Hq.call(this,t);},N(26,114,fi,e.tw,e.uw,e.vw),U.Yb=function(t){e.aw(this,t);},U.$b=function(t){e.bw(this,t);},U.c=false,l.j8=O(26),e.zw=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;for(u=new y.Mj(),e.bw(A,1),Q=j(l.Feb,ht,6,A.f,16,1),n=0;n<A.f;n++)if(t.f[n]&&(e.Tl(t.i,n)==7||e.Tl(t.i,n)==8||e.Tl(t.i,n)==16||e.Tl(t.i,n)==34||e.Tl(t.i,n)==52)){for(Q[n]=true,o=0;o<A.j[n];o++)if(P=A.i[n][o],i=A.k[n][o],s=A.n[n][o],A.o[P]!=0&&A.o[n]<s){for(Q[P]=true,y.sj(u,T.Aib(i)),L=0;L<A.j[P];L++)r=A.i[P][L],Q[r]||(Q[r]=true,D=A.k[P][L],h=A.n[P][L],A.o[P]+2==s+h&&(y.sj(u,T.Aib(D)),s>=h&&(A.o[r]<h?e.Iw(A,r)&&e.Cw(t,A,r,true,false,Q,u):e.Cw(t,A,r,true,true,Q,u)),h>=s&&e.Iw(A,n)&&e.Cw(t,A,r,false,false,Q,u),t.f[r]&&(e.Tl(t.i,r)==5||e.Tl(t.i,r)==6||e.Tl(t.i,r)==7||e.Tl(t.i,r)==8||e.Tl(t.i,r)==16||e.Tl(t.i,r)==34||e.Tl(t.i,r)==52)&&A.o[r]<h&&(s<=2&&h>=2&&e.Iw(A,n)&&e.Aw(t,A,i,D),s>=2&&h<=2&&e.Iw(A,r)&&e.Aw(t,A,D,i)),y.Ej(u,u.a.length-1)),Q[r]=false);y.Ej(u,u.a.length-1),Q[P]=false;}Q[n]=false;}},e.Aw=function(t,A,n,P){var r;r=new e.Ow(t,A),e.Nw(r,n,e.fm(A,n)==1?2:3),e.Nw(r,P,e.fm(A,P)==2?1:2),t.g[n]=true,t.g[P]=true,y.Swb(t.e,r)&&y.Upb(t.d,r);},e.Bw=function(t,A){y.Swb(t.e,A)&&y.Upb(t.d,A);},e.Cw=function(t,A,n,P,r,i,D){var u,o,Q,L,s,h,b,c,g,B,k,M;for(b=0;b<A.j[n];b++)if(u=A.i[n][b],!i[u]&&(L=A.k[n][b],k=e.fm(A,L),P&&k>=2||!P&&k<=2)){for(i[u]=true,D.add(T.Aib(L)),c=0;c<A.j[u];c++)if(o=A.i[u][c],!i[o]&&(s=A.k[u][c],M=e.fm(A,s),A.o[u]+2==k+M&&(P&&M<=2||!P&&M>=2))){if(i[o]=true,D.add(T.Aib(s)),t.f[o]&&(e.Tl(t.i,o)==5||e.Tl(t.i,o)==6||e.Tl(t.i,o)==7||e.Tl(t.i,o)==8||e.Tl(t.i,o)==16||e.Tl(t.i,o)==34||e.Tl(t.i,o)==52)&&(!P||e.Iw(A,o))){for(h=new e.Ow(t,A),g=0;g<D.a.length;g++)Q=(w.Pzb(g,D.a.length),D.a[g]).a,B=g<2?P^(g&1)==0:r^(g&1)==0,B?e.Nw(h,Q,e.fm(A,Q)==1?2:3):e.Nw(h,Q,e.fm(A,Q)==2?1:2),t.g[Q]=true;y.Swb(t.e,h)&&y.Upb(t.d,h);}else e.Cw(t,A,o,P,r,i,D);y.Ej(D,D.a.length-1),i[o]=false;}y.Ej(D,D.a.length-1),i[u]=false;}return  false;},e.Dw=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(t.b=j(l.j7,H,6,t.i.f,15,1),u=j(l.j7,H,6,t.i.f,15,1),P=j(l.Feb,ht,6,t.i.g,16,1),s=0,n=0;n<t.i.g;n++)if(!P[n]&&t.g[n])for(++s,t.b[e.Zl(t.i,0,n)]=s,t.b[e.Zl(t.i,1,n)]=s,P[n]=true,Q=0;Q<2;Q++)for(A=e.Zl(t.i,Q,n),t.b[A]=s,D=0,o=0,u[0]=A;D<=o;){for(L=0;L<e.$o(t.i,u[D]);L++)i=e._o(t.i,u[D],L),!P[i]&&t.g[i]&&(P[i]=true,r=e.Zo(t.i,u[D],L),t.b[r]==0&&(t.b[r]=s,u[++o]=r));++D;}t.j=s;},e.Ew=function(t){var A,n,P;for(t.n=j(l.j7,H,6,t.j,15,1),t.k=j(l.j7,H,6,t.j,15,1),t.o=j(l.j7,H,6,t.j,15,1),A=0;A<t.i.f;A++)t.b[A]!=0&&(P=t.b[A]-1,t.a!=null&&(t.k[P]+=t.a[A]),t.c!=null&&(t.o[P]+=t.c[A]));for(n=0;n<t.i.g;n++)t.g[n]&&e.fm(t.i,n)==2&&(t.n[t.b[e.Zl(t.i,0,n)]-1]+=2);},e.Fw=function(t){var A,n,P,r;for(P=0;P<t.i.r;P++)for(r=0;r<2;r++)A=e.Zl(t.i,r,P),n=e.Zl(t.i,1-r,P),e.Tl(t.i,A)==1&&e.Ml(t.i,A)>1&&e.Tl(t.i,n)>1&&t.b[n]!=0&&(e.Ml(t.i,A)==2?(t.a==null&&(t.a=j(l.j7,H,6,t.i.q,15,1)),++t.a[n]):(t.c==null&&(t.c=j(l.j7,H,6,t.i.q,15,1)),++t.c[n]),e.Um(t.i,A));(t.a!=null||t.c!=null)&&e.vl(t.i);},e.Gw=function(t){var A;for(t.e=new y.Xwb(),t.d=new y.eqb(),e.Bw(t,new e.Ow(t,t.i)),A=e.fw(t.i);!y.Ypb(t.d);)if(e.Mw(y.$pb(t.d),A),e.zw(t,A),t.e.a.c>=e.yw){T.Zjb(),""+e.rg(new e.Ng(t.i));break;}},e.Hw=function(t){var A,n,P,r,i,D,u,o,Q,L;if(t.e.a.c==1)return t.i;for(Q=e.fw(t.i),e.On(Q,true),e.bw(Q,7),r=0;r<Q.g;r++)t.g[r]&&(Q.H[r]=1,Q.T=0,e.Ln(Q,r,3,true));for(n=0;n<Q.f;n++)t.b[n]!=0&&e.tp(t.i,n)<4&&(e.Bo(Q,n),Q.u[n]&=tP,Q.T&=7,e.on(Q,n,0,-1));for(u=j(l.j7,H,6,t.j,15,1),o=j(l.j7,H,6,t.j,15,1),P=new e.Ng(Q).f,A=0;A<Q.f;A++)t.b[A]!=0&&(L=t.b[A]-1,o[L]<P[A]&&(o[L]=P[A],u[L]=A));for(i=0;i<t.j;i++)D=""+t.n[i]+"|"+t.k[i]+"|"+t.o[i],e.mn(Q,u[i],D);return Q;},e.Iw=function(t,A){return t.e[A]-e.tp(t,A)+e.op(t,A)<=0?false:e.Hm(t,A)?true:t.o[A]==0;},e.Jw=function(t){var A,n,P,r,i,D,u,o;e.bw(t.i,1),u=t.i.f;do--u;while(u>=0&&e.Tl(t.i,u)==1);for(A=0;A<u;A++)if(e.Tl(t.i,A)==1){e.Zn(t.i,A,u);do--u;while(e.Tl(t.i,u)==1);}if(u!=t.i.f-1){for(D=j(l.Feb,ht,6,t.i.g,16,1),i=0;i<t.i.g;i++)n=e.Zl(t.i,0,i),P=e.Zl(t.i,1,i),(e.Tl(t.i,n)==1||e.Tl(t.i,P)==1)&&(D[i]=true);o=t.i.g;do--o;while(o>=0&&D[o]);for(r=0;r<o;r++)if(D[r]){e.$n(t.i,r,o),D[r]=false;do--o;while(D[o]);}}},e.Kw=function(t){var A,n;for(this.i=e.fw(t),e.Jw(this),e.bw(this.i,7),this.g=j(l.Feb,ht,6,this.i.g,16,1),this.f=j(l.Feb,ht,6,this.i.f,16,1),A=0;A<this.i.f;A++)n=e.mo(e.Tl(this.i,A))[0],this.f[A]=e.tp(this.i,A)<n;e.Gw(this),e.Dw(this),e.Fw(this),e.Ew(this);},N(241,1,{},e.Kw),U.j=0,e.yw=uP,l.l8=O(241),e.Lw=function(t,A){return Lt.$3(t.a,A.a);},e.Mw=function(t,A){var n,P;for(P=0;P<t.b.i.g;P++)t.b.g[P]&&(n=3&t.a[P>>4]>>2*(P&15),e.Mn(A,P,n==1?1:n==2?t.b.g[P]&&!e.jq(t.b.i,P)?386:2:n==3?4:16));},e.Nw=function(t,A,n){var P,r;P=A>>4,r=2*(A&15),t.a[P]&=~(3<<r),t.a[P]|=n<<r;},e.Ow=function(t,A){var n;for(this.b=t,this.a=j(l.j7,H,6,(t.i.g+15)/16|0,15,1),n=0;n<t.i.g;n++)this.a[n>>4]|=d.Math.min(3,e.fm(A,n))<<2*(n&15);},N(166,1,{166:1,32:1},e.Ow),U.Sb=function(t){return e.Lw(this,t);},l.k8=O(166),e.Qw=function(t,A,n){var P,r,i,D,u;for(r=0,D=new y.Bqb(n);D.a<D.c.a.length;)i=y.Aqb(D),w.aAb(i).length!=0&&(u=C.S2(A,i).b,r<u&&(r=u));return P=t.c*yn*(n.a.length-1)+t.c,new Tt.KN(t.i[0].a,t.i[0].b-t.c/2,r,P);},e.Rw=function(t,A,n){return !!t.b&&Tt.FN(t.b,A,n);},e.Sw=function(t){var A,n,P,r,i;if(n=T.rjb(t,Y1(38)),n==-1)return t;for(r=0,i=new T.Mjb();n!=-1;)T.Ljb(i,(w.Wzb(r,n,w.aAb(t).length),w.aAb(t).substr(r,n-r))),w.Xzb(n+1,w.aAb(t).length),w.aAb(t).charCodeAt(n+1)==38?(i.a+="&",r=n+2):(A=(w.Xzb(n+1,w.aAb(t).length),w.aAb(t).charCodeAt(n+1)),A=A-(A<65?48:A<97?65:97),P=(w.Xzb(n+2,w.aAb(t).length),w.aAb(t).charCodeAt(n+2)),P=P-(P<65?48:P<97?65:97),T.Jjb(i,16*A+P&n1),r=n+3),n=T.sjb(t,Y1(38),r);return T.Ljb(i,(w.Xzb(r,w.aAb(t).length+1),w.aAb(t).substr(r))),i.a;},e.Tw=function(t){var A,n;for(n=new T.Mjb(),A=0;A<w.aAb(t).length;A++)switch(w.Xzb(A,w.aAb(t).length),w.aAb(t).charCodeAt(A)){case 38:n.a+="&&";break;case 9:n.a+="&09";break;case 10:n.a+="&0A";break;case 32:n.a+="&20";break;default:T.Jjb(n,(w.Xzb(A,w.aAb(t).length),w.aAb(t).charCodeAt(A)));}return n.a;},e.Uw=function(t,A){var n;return n=e.Vw(t),e.Qw(t,A,n);},e.Vw=function(t){var A,n,P;if(n=new y.Mj(),A=T.rjb(t.e,Y1(10)),A==-1)y.sj(n,t.e);else {for(P=0;A!=-1;)y.sj(n,T.Bjb(t.e,P,A)),P=A+1,A=T.qjb(t.e,10,P);y.sj(n,T.Ajb(t.e,P));}return n;},e.Ww=function(t,A,n){t.i[0].a=A,t.i[0].b=n;},e.Xw=function(t,A){try{t.c=N0(A);}catch(n){if(n=M1(n),!Kt(n,40))throw Ut(n);}},e.Yw=function(t,A){try{t.d=T.Whb(A,P1,St);}catch(n){if(n=M1(n),!Kt(n,40))throw Ut(n);}},e.Zw=function(t,A,n,P){t.e=A,t.c=n,t.d=P;},e.$w=function(t,A){try{t.i[0].a=N0(A);}catch(n){if(n=M1(n),!Kt(n,40))throw Ut(n);}},e._w=function(t,A){try{t.i[0].b=N0(A);}catch(n){if(n=M1(n),!Kt(n,40))throw Ut(n);}},e.ax=function(){e.cx.call(this,new Tt.vN());},e.bx=function(t){var A,n,P,r;for(e.ax.call(this),A=0;A!=-1&&(n=w.aAb(t).indexOf('="',A),n!=-1);)P=(w.Wzb(A+1,n,w.aAb(t).length),w.aAb(t).substr(A+1,n-(A+1))),A=w.aAb(t).indexOf('"',n+2),r=A==-1?(w.Xzb(n+1,w.aAb(t).length+1),w.aAb(t).substr(n+1)):(w.Wzb(n+1,A,w.aAb(t).length),w.aAb(t).substr(n+1,A-(n+1))),T.kjb(P,"text")?this.e=e.Sw(r):T.kjb(P,"x")?e.$w(this,r):T.kjb(P,"y")?e._w(this,r):T.kjb(P,"size")?e.Xw(this,r):T.kjb(P,"style")&&e.Yw(this,r);},e.cx=function(t){this.e="",this.c=9,this.d=0,this.i=j(l.fab,C1,55,1,0,1),this.i[0]=t;},N(94,193,{193:1,94:1},e.ax,e.bx),U.Eb=function(t,A){return this.a=!!this.b&&Tt.FN(this.b,t,A),this.a;},U.Fb=function(){this.a=false;},U.Gb=function(){var t;return t=new e.ax(),e.Zw(t,this.e,this.c,this.d),e.Ww(t,this.i[0].a,this.i[0].b),t.g=this.g,t;},U.Hb=function(t,A){return e.Rw(this,t,A);},U.Ib=function(t,A){var n,P,r;for(P=A?A.c*this.c:this.c,C.Y2(t,Rt(d.Math.round(P)),(this.d&1)!=0,(this.d&2)!=0),C.$2(t,this.g?fA:C.X2(t)?-1:O0),r=e.Vw(this),this.b=e.Qw(this,t,r),A&&e.xi(A,this.b),n=0;n<r.a.length;n++)C.N2(t,this.b.c,this.b.d+1+P*5/6+P*yn*n,(w.Pzb(n,r.a.length),r.a[n]));},U.Jb=function(t){return e.Uw(this,t);},U.Kb=function(){var t;return t=new T.Ujb(),T.Sjb(t,' text="'+e.Tw(this.e)+'"'),T.Sjb(t,' x="'+this.i[0].a+'"'),T.Sjb(t,' y="'+this.i[0].b+'"'),this.c!=9&&T.Sjb(t,' size="'+R.ukb(new R.xkb(this.c,new R.cmb()))+'"'),this.d!=0&&T.Sjb(t,' style="'+this.d+'"'),t.a;},U.Lb=function(){return "text";},U.Mb=function(t){var A,n;A=(n=e.Vw(this),e.Qw(this,t,n)),C.$2(t,C.W2(t)),C.Q2(t,A.c,A.d,A.b,A.a);},U.Nb=function(t){e.me(this,t),this.c*=t;},U.Pb=function(t,A){e.qe(this,t,A),this.c=this.f*t;},U.Qb=function(t,A){e.re(this,t,A),this.f=this.c;},U.a=false,U.c=0,U.d=0,U.f=0,l.m8=O(94),e.px=function(t){t.b=new y.Mj(),t.a=new y.Mj();},e.qx=function(t,A){var n,P;return n=e.Xv(t,A),n==-1?-1:(P=t.b.a.length,y.sj(t.b,A),y.rj(t.a,n,T.Aib(P)),P);},e.rx=function(){e.Zv.call(this),e.px(this);},N(282,102,ii,e.rx),U.bc=function(t){var A;return A=e.Yv(this,t),A==-1?-1:y.xj(this.a,A).a;},l.n8=O(282),x.tx=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt;for(Dt=e.Ep(t.e),lt=y.xj(Dt.k,A).length,Qt=y.xj(Dt.j,A),ut=y.xj(Dt.k,A),z=true,k=1;k<lt;k++)if(x.Ix(t.b,ut[k])!=x.Ix(t.b,ut[0])){z=false;break;}if(z){for(i=(lt-2)*Se/lt,M=0;M<lt;M++)x.yx(t,Qt[M],ut[M],ut[M==0?lt-1:M-1],i);return;}for(G=j(l.i7,w0,6,lt,15,1),r=0,p=0;p<lt;p++)G[p]=e.Qo(t.e,Qt[p])==0?1.9106119386631804:e.Qo(t.e,Qt[p])==1?2.094499897810209:Se,r+=G[p];for(P=((lt-2)*Se-r)/lt,q=0;q<lt;q++)G[q]+=P;for(L=j(l.i7,w0,6,lt,15,1),S=1;S<lt;S++)L[S]=L[S-1]+Se-G[S];for(u=j(l.i7,w0,6,lt,15,1),D=0;D<100;D++){for(ct=0,yt=0,M=0;M<lt;M++)ct+=x.Ix(t.b,ut[M])*d.Math.sin(L[M]),yt+=x.Ix(t.b,ut[M])*d.Math.cos(L[M]);for(b=e.no(0,0,ct,yt),g=d.Math.sqrt(ct*ct+yt*yt),ot=-1,tt=0,V=0;V<lt;V++)I=V==0?lt-1:V-1,Z=V+1==lt?0:V+1,o=e.oo(L[V],L[I]),Q=e.oo(L[Z],L[V]),it=e.oo(G[V],G[Z]),u[V]=2*o-2*Q+2*it,c=d.Math.cos(L[V]-wr-b),h=g*c-Di*u[V],d.Math.abs(h)>d.Math.abs(tt)&&(tt=h,ot=V);s=d.Math.exp(-5*D/100),L[ot]+=s*tt;}for(B=0;B<lt;B++)I=B==0?lt-1:B-1,n=L[I]+Se-L[B],n>I1&&(n-=I1),x.yx(t,Qt[B],ut[I],ut[B],n);},x.ux=function(t,A){var n,P,r,i,D,u,o,Q,L;for(o=e.Ep(t.e),Q=y.xj(o.k,A).length,u=y.xj(o.k,A),L=false,r=0;r<Q;r++)if(t.f[u[r]]==0){L=true;break;}if(L){for(n=(180*Q-360)/Q*.01745329238474369,D=y.xj(o.j,A),i=0;i<Q;i++)x.yx(t,D[i],u[i==0?Q-1:i-1],u[i],n);if(Q<=4)for(P=0;P<Q;P++)t.f[u[P]]+=Q;}},x.vx=function(t,A,n,P){var r,i,D,u;for(i=-1,D=-1,u=0;u<e.Lo(t.e,A);u++)if(r=e.Zo(t.e,A,u),r==n||r==P)if(D==-1)D=u;else {i=u;break;}return t.a[A][i][D];},x.wx=function(t,A,n,P){return n<P?t.a[A][P][n]:t.a[A][n][P];},x.xx=function(t,A){var n,P,r,i,D,u,o;for(P=j(l.Feb,ht,6,e.$o(t.e,A),16,1),o=0,r=0;r<e.$o(t.e,A);r++){for(u=0,D=-1,i=0;i<e.$o(t.e,A);i++)P[i]||(n=e._o(t.e,A,i),u<t.f[n]&&(u=t.f[n],D=i));if(u==0)return o;o<<=8,o+=u,P[D]=true;}return o;},x.yx=function(t,A,n,P,r){var i,D,u,o;for(i=-1,D=-1,o=0;o<e.Lo(t.e,A);o++)if(u=e._o(t.e,A,o),u==n||u==P)if(i==-1)i=o;else {D=o;break;}t.a[A][D][i]==0&&(t.a[A][D][i]=r,t.d[A]+=r,++t.c[A]);},x.zx=function(t,A,n,P,r){var i,D,u,o,Q,L;for(i=e.Lo(t.e,A),o=e.$o(t.e,A),Q=1;Q<i;Q++)for(D=Q<o?t.f[e._o(t.e,A,Q)]:0,L=0;L<Q;L++)t.a[A][Q][L]==0&&(u=L<o?t.f[e._o(t.e,A,L)]:0,(D==n&&u==P||D==P&&u==n)&&(t.a[A][Q][L]=r));},x.Ax=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I;for(this.b=A,this.e=t,this.a=j(l.i7,c0,456,this.e.f,0,3),r=0;r<this.e.f;r++)for(this.a[r]=j(l.i7,{456:1,4:1,9:1,5:1,8:1},46,e.Lo(this.e,r),0,2),o=1;o<e.Lo(this.e,r);o++)this.a[r][o]=j(l.i7,w0,6,o,15,1);for(this.c=j(l.j7,H,6,this.e.f,15,1),this.d=j(l.i7,w0,6,this.e.f,15,1),this.f=j(l.j7,H,6,this.e.g,15,1),V=e.Ep(this.e),h=j(l.Feb,ht,6,V.j.a.length,16,1),e.at(V,h,j(l.Feb,ht,6,V.j.a.length,16,1),j(l.j7,H,6,V.j.a.length,15,1),true),u=3;u<=7;u++)for(k=0;k<V.j.a.length;k++)I=y.xj(V.k,k).length,I==u&&(h[k]?x.tx(this,k):I<=4&&x.ux(this,k));for(s=j(l.Feb,ht,6,this.e.f,16,1),B=0;B<V.j.a.length;B++)if(h[B])for(p=y.xj(V.j,B),q=0,S=p.length;q<S;++q)M=p[q],s[M]=true;for(i=v(F(l.j7,1),H,6,15,[0,0,1,3,6,10,15,21]),P=0;P<this.e.f;P++){if(D=e.Lo(this.e,P),D>4){for(o=1;o<D;o++)for(c=0;c<o;c++)this.a[P][o][c]=wr;this.c[P]=i[D];continue;}if(this.c[P]!=i[D])if(e.iq(this.e,P)&&(s[P]||e.Uo(this.e,P)<=4&&e.Qo(this.e,P)>0)){if(D>2){if(this.c[P]==1)e.Tl(this.e,P)<=14?n=(ui-this.d[P])/2:n=oi+(oi-this.d[P])*.18000000715255737;else if(n=ui-this.d[P],D>3){if(this.c[P]==2){for(b=j(l.Feb,ht,6,e.Lo(this.e,P),16,1),Q=1;Q<D;Q++)for(g=0;g<Q;g++)this.a[P][Q][g]!=0&&(b[Q]=!b[Q],b[g]=!b[g]);for(L=0;L<D;L++)if(b[L]){for(g=L+1;g<D;g++)if(b[g]){this.a[P][g][L]=n;break;}break;}}n=wr;}for(o=1;o<D;o++)for(c=0;c<o;c++)this.a[P][o][c]==0&&(this.a[P][o][c]=n);}}else if(e.iq(this.e,P)&&e.Uo(this.e,P)<=4)switch(x.xx(this,P)){case 771:x.zx(this,P,0,3,2.0653998851776123),x.zx(this,P,0,0,1.9814722631346626);break;case 1028:x.zx(this,P,0,4,1.9797999858856201),x.zx(this,P,0,0,1.94691481878138);break;case 393987:x.zx(this,P,0,3,li),x.zx(this,P,0,6,li),x.zx(this,P,3,3,1.7229016938441077);break;case 459779:x.zx(this,P,0,3,gr),x.zx(this,P,0,4,gr),x.zx(this,P,0,7,gr),x.zx(this,P,3,4,1.9322539839360076);break;case 525316:x.zx(this,P,0,4,Qi),x.zx(this,P,0,8,Qi),x.zx(this,P,4,4,1.99944913298566);case 394758:x.zx(this,P,0,6,2.526099920272827);break;case 460550:x.zx(this,P,0,6,Li),x.zx(this,P,0,7,Li);break;case 526087:x.zx(this,P,0,7,Ui),x.zx(this,P,0,8,Ui);break;case 526344:x.zx(this,P,0,8,2.186300039291382);break;case 50529027:x.zx(this,P,3,3,2.4189000129699707);break;case 67371779:x.zx(this,P,3,4,2.2298998832702637);break;case 67372036:x.zx(this,P,4,4,2.094399929046631);break;case 101057283:x.zx(this,P,3,6,1.839926051241747),x.zx(this,P,3,3,2.9061476191098734);break;case 117834755:x.zx(this,P,3,4,2.812249087174905),x.zx(this,P,3,7,1.7910569124592968),x.zx(this,P,4,6,2.1224948975613245);break;case 134677507:x.zx(this,P,3,4,2.642428360523752),x.zx(this,P,3,8,2.027723514585844),x.zx(this,P,4,7,2.251474717631936);break;case 117900035:x.zx(this,P,3,7,2.109753935530918),x.zx(this,P,3,3,3.1052897491356646);break;case 117900292:x.zx(this,P,4,7,2.090729910747413),x.zx(this,P,4,4,2.551671293386306);break;case 134743044:x.zx(this,P,4,8,2.139250042271712),x.zx(this,P,4,4,2.3520055858942612);}else for(n=e.Tl(this.e,P)>10?yi:e.Qo(this.e,P)==2?Se:e.$p(this.e,P,true)?si:e.Qo(this.e,P)==0?yi:si,o=1;o<D;o++)for(c=0;c<o;c++)this.a[P][o][c]=n;}},N(374,1,{},x.Ax),l.o8=O(374),x.Hx=function(){x.Hx=D1,x.Fx=v(F(l.Feb,1),ht,6,16,[false,false,false,false,false,true,true,true,true,false,false,false,false,false,false,true,true]);},x.Ix=function(t,A){return t.a[A];},x.Jx=function(t){x.Hx();var A,n;for(e.bw(t,7),this.a=j(l.i7,w0,6,t.r,15,1),this.b=j(l.i7,w0,6,t.r,15,1),A=0;A<t.r;A++)n=x.Nx(t,A),n==-1?(this.a[A]=x.Ox(t,A),this.b[A]=x.Px(t,A)):(this.a[A]=n==-1?2.000499963760376:x.Dx[n],this.b[A]=n==-1?1:x.Ex[n]);},x.Kx=function(t,A){return A>=t.f?0:A<t.f&&e.ot(t.p,A)&&t.C[A]==6&&t.s[A]!=0?1:t.o[A];},x.Lx=function(t,A,n,P,r,i,D,u,o){var Q,L,s,h,b;return h=P<x.Fx.length&&x.Fx[P]?d.Math.min(3,i):0,b=r<x.Fx.length&&x.Fx[r]?d.Math.min(3,D):0,u=d.Math.min(7,u),o=d.Math.min(7,o),Q=u+(h<<3)+(d.Math.min(127,P)<<5),L=o+(b<<3)+(d.Math.min(127,r)<<5),s=n?8:A?8+t:t,(s<<24)+(Q<L?(Q<<12)+L:(L<<12)+Q);},x.Mx=function(t,A,n,P,r,i,D,u,o,Q){var L;return L=x.Rx(x.Lx(t,A,n,P,r,i,D,u,o)),!Q&&L==-1&&(L=x.Rx(x.Lx(t,A,n,P,r,i,D,7,7)),L==-1&&(L=x.Rx(x.Lx(t,A,n,P,r,3,3,7,7)))),L;},x.Nx=function(t,A){var n,P,r,i,D,u;return n=t.D[0][A],P=t.D[1][A],r=t.C[n],i=t.C[P],D=t.j[n],u=t.j[P],x.Mx(e.fm(t,A),A<t.g&&e.pt(t.p,A),A<t.g&&(e.ut(t.p,A)||t.H[A]==8),r,i,x.Kx(t,n),x.Kx(t,P),D,u,false);},x.Ox=function(t,A){var n,P;return n=e.Tl(t,t.D[0][A]),P=e.Tl(t,t.D[1][A]),(n<(x.qz(),x.oz).length?x.oz[n]:n<x.pz.length?sn*x.pz[n]:oP)+(P<x.oz.length?x.oz[P]:P<x.pz.length?sn*x.pz[P]:oP);},x.Px=function(t,A){var n,P;return n=e.Tl(t,t.D[0][A]),P=e.Tl(t,t.D[1][A]),(n<(x.qz(),x.oz).length?jr:.125)+(P<x.oz.length?jr:.125);},x.Qx=function(){var t,A,n,P,r,i,D;if(!x.Gx)try{for(t=x.Ey("bondLengthData.txt"),n=Ct.ngb(t),A=n==null?0:T.Whb(n,P1,St),x.Cx=j(l.j7,H,6,A,15,1),x.Dx=j(l.i7,w0,6,A,15,1),x.Ex=j(l.i7,w0,6,A,15,1),x.Bx=j(l.j7,H,6,A,15,1),r=0;r<A;r++)if(D=Ct.ngb(t),D!=null&&(i=T.yjb(D,"\\t"),i.length==4))try{x.Cx[r]=T.Whb(i[0],P1,St),x.Dx[r]=N0(i[1]),x.Ex[r]=N0(i[2]),x.Bx[r]=T.Whb(i[3],P1,St);}catch(u){if(u=M1(u),Kt(u,40))break;throw Ut(u);}x.Gx=!0;}catch(u){if(u=M1(u),Kt(u,75))P=u,T.MB(P,(T.Zjb(),T.Yjb),"");else throw Ut(u);}},x.Rx=function(t){var A,n,P,r;for(x.Gx||x.Qx(),r=Z1,P=j1,n=0;n<13;n++){if(A=r>=x.Cx.length||t<x.Cx[r]?-1:t==x.Cx[r]?0:1,A==0)return r;r=A<0?r-P:r+P,P=P/2|0;}return  -1;},N(373,1,{},x.Jx),x.Gx=false,l.p8=O(373),x.Sx=function(t,A){t.q=x.Ry(new x.Uy(t.o,A),t);},x.Tx=function(t,A){var n,P,r,i,D,u,o,Q,L;return n=x.Zx(t,A[0]),P=x.Zx(t,A[1]),r=x.Zx(t,A[2]),i=x.Zx(t,A[3]),o=new e.oi(P.a-n.a,P.b-n.b,P.c-n.c),Q=new e.oi(r.a-P.a,r.b-P.b,r.c-P.c),L=new e.oi(i.a-r.a,i.b-r.b,i.c-r.c),D=new e.oi(o.b*Q.c-o.c*Q.b,-(o.a*Q.c-o.c*Q.a),o.a*Q.b-o.b*Q.a),u=new e.oi(Q.b*L.c-Q.c*L.b,-(Q.a*L.c-Q.c*L.a),Q.a*L.b-Q.b*L.a),-d.Math.atan2(d.Math.sqrt(Q.a*Q.a+Q.b*Q.b+Q.c*Q.c)*(o.a*u.a+o.b*u.b+o.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},x.Ux=function(t,A){return x.Xx(t,A),x.Fy(t.q,A.q);},x.Vx=function(t,A){var n;for(n=0;n<A.q;n++)e.ji(t.k[n],A.J[n]);},x.Wx=function(t,A){var n;for(n=0;n<A.k.length;n++)e.ji(t.k[n],A.k[n]);A.j==null?t.j=null:t.j=y.Kqb(A.j,A.j.length);},x.Xx=function(t,A){var n;(!t.q||!A.q)&&(n=new x.Ty(t.o),!t.q&&(t.q=x.Ry(n,t)),!A.q&&(A.q=x.Ry(n,A)));},x.Yx=function(t,A){return t.j==null?-1:t.j[A];},x.Zx=function(t,A){return t.k[A];},x.$x=function(t){return t.p==null?t.o.P:t.p;},x._x=function(t,A){return t.k[A].a;},x.ay=function(t,A){return t.k[A].b;},x.by=function(t,A){return t.k[A].c;},x.cy=function(t,A,n){for(t.j==null&&(t.j=j(l.Eeb,Yt,6,t.o.r,15,1),y.Qqb(t.j));n<0;)n=n+360<<16>>16;for(;n>=360;)n=n-360<<16>>16;t.j[A]=n;},x.dy=function(t,A,n){e.ji(t.k[A],n);},x.ey=function(t,A,n){t.k[A]=n;},x.fy=function(t,A){t.p=A;},x.gy=function(t,A,n){t.k[A].a=n;},x.hy=function(t,A,n){t.k[A].b=n;},x.iy=function(t,A,n){t.k[A].c=n;},x.jy=function(t,A){var n;for(!A&&(A=e.fw(t.o)),A.T&=-249,n=0;n<A.q;n++)e.ji(A.J[n],t.k[n]);return t.p!=null&&e.Un(A,t.p),A;},x.ky=function(t){var A;for(this.o=t,this.k=j(l.C7,qe,29,t.q,0,1),A=0;A<t.q;A++)this.k[A]=new e.pi(t.J[A]);this.n=NaN;},x.ly=function(t){x.my.call(this,t,t.o);},x.my=function(t,A){var n;for(this.o=A,this.k=j(l.C7,qe,29,t.k.length,0,1),n=0;n<this.k.length;n++)this.k[n]=new e.pi(t.k[n]);t.j!=null&&(this.j=y.Kqb(t.j,t.j.length)),this.p=t.p==null||T.jjb(t.p)?t.p:T.gjb(t.p,ve),this.n=NaN;},N(63,1,{63:1,32:1},x.ly),U.Sb=function(t){return x.Ux(this,t);},U.n=0,l.q8=O(63),x.qy=function(t,A){var n,P,r,i,D;return A==null?null:(D=y.opb(t.c,A),D||(P=w.aAb("<").length,(T.kjb(w.aAb(A).substr(w.aAb(A).length-P,P),"<")||(r=w.aAb("-").length,T.kjb(w.aAb(A).substr(w.aAb(A).length-r,r),"-")))&&(D=y.opb(t.c,(i=w.aAb("<").length,T.kjb(w.aAb(A).substr(w.aAb(A).length-i,i),"<")?T.Bjb(A,0,w.aAb(A).length-1)+">":(n=w.aAb("-").length,T.kjb(w.aAb(A).substr(w.aAb(A).length-n,n),"-")?T.Bjb(A,0,w.aAb(A).length-1)+"+":A))),D)?(D=new x.mz(D),y.Zvb(t.c,A,D),D):null));},x.ry=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q;t.b|=A;try{for(p=x.Ey("torsionID.txt"),P=(A&1)==0?null:x.Ey("torsionAngle.txt"),h=(A&1)==0?null:x.Ey("torsionRange.txt"),u=(A&1)==0?null:x.Ey("torsionFrequency.txt"),i=(A&2)==0?null:x.Ey("torsionBins.txt"),q=Ct.ngb(p);q!=null;){if(M=y.opb(t.c,q),M||(M=new x.lz((c=w.aAb("<").length,T.kjb(w.aAb(q).substr(w.aAb(q).length-c,c),"<")||(g=w.aAb(">").length,T.kjb(w.aAb(q).substr(w.aAb(q).length-g,g),">"))?0:(B=w.aAb("-").length,T.kjb(w.aAb(q).substr(w.aAb(q).length-B,B),"-")||(k=w.aAb("+").length,T.kjb(w.aAb(q).substr(w.aAb(q).length-k,k),"+"))?1:(b=w.aAb("=").length,T.kjb(w.aAb(q).substr(w.aAb(q).length-b,b),"=")?3:2)))),y.Zvb(t.c,q,M)),P)for(n=T.yjb(Ct.ngb(P),","),M.a=j(l.Eeb,Yt,6,n.length,15,1),Q=0;Q<n.length;Q++)M.a[Q]=T.Whb(n[Q],Zn,32767)<<16>>16;if(h)for(s=T.yjb(Ct.ngb(h),","),M.d=y1(l.Eeb,[P0,Yt],[16,6],15,[s.length,2],2),Q=0;Q<s.length;Q++)L=T.qjb(s[Q],45,1),M.d[Q][0]=T.Whb(T.Bjb(s[Q],0,L),Zn,32767)<<16>>16,M.d[Q][1]=T.Whb(T.Ajb(s[Q],L+1),Zn,32767)<<16>>16;if(u)for(o=T.yjb(Ct.ngb(u),","),M.c=j(l.Eeb,Yt,6,o.length,15,1),Q=0;Q<o.length;Q++)M.c[Q]=T.Whb(o[Q],-128,127)<<24>>24;if(i)for(r=T.yjb(Ct.ngb(i),","),M.b=j(l.f7,Y,6,r.length,15,1),Q=0;Q<r.length;Q++)M.b[Q]=T.Whb(r[Q],-128,127)<<24>>24;q=Ct.ngb(p);}}catch(S){if(S=M1(S),Kt(S,75))D=S,T.MB(D,(T.Zjb(),T.Yjb),"");else throw Ut(S);}},x.sy=function(t){var A,n,P,r;for(t.a=15,r=(n=new y.Oob(t.c).a.bd().Tb(),new y.Vob(n));r.a.dd();)P=(A=r.a.ed(),A.hd()),x.jz(P);},x.ty=function(){this.c=new y.fwb(),this.b=0;},x.uy=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(P=j(l.h7,Ft,6,2,15,1),b=x.xy(t,n),o=0;o<2;o++)if(n[3*o]!=-1){for(i=2-o,c=3-3*o,Q=0,h=0;h<3;h++)D=e.Zo(t,n[i],h),D!=b[1-o]&&t.C[D]!=1&&(n[c]=D,A?P[Q++]=x.Tx(A,n):P[Q++]=e.el(t,n));return n[c]=-1,x.wy(P);}for(r=j(l.h7,Ft,6,2,15,1),L=0,u=0;u<3;u++)if(g=e.Zo(t,n[1],u),g!=b[0]&&t.C[g]!=1){for(n[0]=g,s=0,h=0;h<3;h++)B=e.Zo(t,n[2],h),B!=b[1]&&t.C[B]!=1&&(n[3]=B,A?r[s++]=x.Tx(A,n):r[s++]=e.el(t,n));P[L++]=x.wy(r);}return n[0]=-1,n[3]=-1,x.wy(P);},x.vy=function(t,A){return A[0]!=-1&&A[3]!=-1?x.Tx(t,A):x.uy(t.o,t,A);},x.wy=function(t){var A,n;return n=(t[1]+t[0])/2,A=d.Math.abs(t[1]-t[0]),A>A1?n:n<0?n+A1:n-A1;},x.xy=function(t,A){var n,P,r;return r=j(l.j7,H,6,2,15,1),e.Xo(t,A[1],A[2])!=-1?(r[0]=A[2],r[1]=A[1]):(n=j(l.j7,H,6,16,15,1),P=e.yp(t,n,A[1],A[2],15,null,null),r[0]=n[1],r[1]=n[P-1]),r;},x.yy=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;for(e.bw(t,7),Q=0,r=0;r<t.g;r++)e.fm(t,r)==1&&!(r<t.g&&e.pt(t.p,r))&&e.tp(t,t.D[0][r])>1&&e.tp(t,t.D[1][r])>1&&(t.F[r]&64)==0&&!((t.F[r]&128)!=0&&(t.p&&r<t.g?e.ft(t.p,r):0)<=5)&&(A[r]=true,++Q);for(D=j(l.j7,H,6,2,15,1),b=j(l.j7,H,6,2,15,1),i=null,P=0;P<t.g;P++)if(A[P]&&(i==null||!i[P])){for(n=0,s=0;s<2;s++)for(D[s]=t.D[s][P],b[s]=t.D[1-s][P],g=b[s];e.Qo(t,D[s])==2&&e.tp(t,D[s])==2&&e.Tl(t,D[s])<10;){for(h=0;h<2;h++)if(u=e.Zo(t,D[s],h),u!=b[s]){o=e._o(t,D[s],h),A[o]&&e.fm(t,o)==1&&(A[o]=false,--Q),b[s]=D[s],D[s]=u,++n;break;}if(D[s]==g){n=0;break;}}n!=0&&(A[P]=false,--Q,e.tp(t,D[0])>1&&e.tp(t,D[1])>1&&(B=e.Lp(t,b[0],D[0]),k=e.Lp(t,b[1],D[1]),L=B<k?0:1,c=e.Xo(t,b[L],D[L]),i==null&&(i=j(l.Feb,ht,6,t.g,16,1)),i[c]=true,A[c]=true,++Q));}return Q;},x.zy=function(t){var A;return A=x.qy(x.py,t),A?x.gz(A):null;},x.Ay=function(t,A,n,P){return x.az(P,t,A),n!=null&&(n[0]=P.f[0],n[1]=P.b[0],n[2]=P.b[1],n[3]=P.f[1]),P.d;},x.By=function(t){var A;return A=x.qy(x.py,t),A?x.hz(A):null;},x.Cy=function(t){var A;return A=x.qy(x.py,t),A?x.fz(A):null;},x.Dy=function(){var t;!x.py&&(x.py=new x.ty()),(x.py.b&1)!=0&&x.py.a!=15&&T.Zjb(),t=1&~x.py.b,t!=0&&(x.ry(x.py,t),t==1&&x.sy(x.py));},x.Ey=function(t){var A,n,P;if(x.oy==null){if(A=_t.mAb("/resources/csd/"+t),A)return x.oy="csd/",new Ct.ogb(new Ct.tgb((P=A,S1.Mmb(),P)));x.oy="cod/";}return new Ct.ogb(new Ct.tgb((n=_t.mAb("/resources/"+x.oy+t),S1.Mmb(),n)));},N(365,1,{},x.ty),U.a=0,U.b=0,l.r8=O(365),x.Fy=function(t,A){var n,P;for(P=0;P<t.b.length;P++)if(n=d.Math.abs(t.b[P]-A.b[P]),n>lP&&n<t.a[P]-lP)return n<t.a[P]/2^t.b[P]<A.b[P]?1:-1;return 0;},x.Gy=function(t,A){var n,P;for(P=0;P<t.b.length;P++)if(n=d.Math.abs(t.b[P]-A.b[P]),n>lP&&n<t.a[P]-lP)return  false;return  true;},x.Hy=function(t,A){this.b=t,this.a=A;},N(256,1,{256:1,32:1},x.Hy),U.Sb=function(t){return x.Fy(this,t);},U.pb=function(){var t,A;for(A=new T.Ujb(),t=0;t<this.b.length;t++)A.a+=t==0?"Torsions: ":", ",T.Sjb(A,(Lt.W3(),Lt.X3(this.b[t],3)+"("+Rt(d.Math.round(this.a[t]+.5))+")"));return A.a;},l.t8=O(256),x.My=function(){x.My=D1,x.Ly=v(F(l.h7,1),Ft,6,15,[I1,A1,Ve,b0]),x.Ky=v(F(l.j7,2),E1,7,0,[v(F(l.j7,1),H,6,15,[0,0,0,2]),v(F(l.j7,1),H,6,15,[0,0,1,2]),v(F(l.j7,1),H,6,15,[0,1,1,3]),v(F(l.j7,1),H,6,15,[2,2,3,2])]);},x.Ny=function(t,A,n,P){var r,i,D,u;if(P==1&&!x.Sy(t,A)){if(e.$o(t.b,A)==3)return  -1;if(e.$o(t.b,A)==3)return r=x.Qy(t,A,n),e.mw(t.b,r[0])==e.mw(t.b,r[1])?r[2]:e.mw(t.b,r[0])==e.mw(t.b,r[2])?r[1]:r[0];}for(D=-1,u=-1,i=0;i<e.$o(t.b,A);i++)r=e.Zo(t.b,A,i),r!=n&&D<e.mw(t.b,r)&&(D=e.mw(t.b,r),u=r);return u;},x.Oy=function(t){var A,n,P,r,i;for(t.a=y1(l.j7,[E1,H],[7,6],15,[t.d.length,4],2),t.c=y1(l.j7,[E1,H],[7,6],15,[t.d.length,2],2),t.e=j(l.j7,H,6,t.d.length,15,1),A=j(l.j7,H,6,2,15,1),r=0;r<t.d.length;r++){for(i=0;i<2;i++)A[0]=e.Zl(t.b,1-i,t.d[r]),A[1]=e.Zl(t.b,i,t.d[r]),x.Yy(t.b,A[1])&&x.Wy(t.b,A[0],A,null),t.a[r][1+i]=A[1],t.c[r][i]=A[0];n=x.Py(t,t.a[r][1],t.c[r][0]),P=x.Py(t,t.a[r][2],t.c[r][1]),t.e[r]=x.Ky[n][P],t.a[r][0]=x.Ny(t,t.a[r][1],t.c[r][0],n),t.a[r][3]=x.Ny(t,t.a[r][2],t.c[r][1],P);}},x.Py=function(t,A,n){var P;if(e.$o(t.b,A)==2)return 1;if(P=x.Qy(t,A,n),e.$o(t.b,A)==3)return e.mw(t.b,P[0])==e.mw(t.b,P[1])?x.Sy(t,A)?2:1:x.Sy(t,A)?1:0;if(e.$o(t.b,A)==4){if(e.mw(t.b,P[0])==e.mw(t.b,P[1])&&e.mw(t.b,P[0])==e.mw(t.b,P[2]))return 3;if(e.mw(t.b,P[0])==e.mw(t.b,P[1])||e.mw(t.b,P[0])==e.mw(t.b,P[2])||e.mw(t.b,P[1])==e.mw(t.b,P[2]))return 1;}return 0;},x.Qy=function(t,A,n){var P,r,i;for(i=0,P=j(l.j7,H,6,e.$o(t.b,A)-1,15,1),r=0;r<e.$o(t.b,A);r++)e.Zo(t.b,A,r)!=n&&(P[i++]=e.Zo(t.b,A,r));return P;},x.Ry=function(t,A){var n,P,r;for(r=j(l.i7,w0,6,t.d.length,15,1),P=j(l.i7,w0,6,t.d.length,15,1),n=0;n<t.d.length;n++)r[n]=x.Xy(x.vy(A,t.a[n]),t.e[n]),P[n]=x.Ly[t.e[n]];return new x.Hy(r,P);},x.Sy=function(t,A){return !!(e.Qo(t.b,A)==1&&e.Tl(t.b,A)<10||e.Tp(t.b,A)||e.$p(t.b,A,true));},x.Ty=function(t){x.My(),this.b=t,e.bw(this.b,63),this.d=x.Vy(t),x.Oy(this);},x.Uy=function(t,A){x.My(),this.b=t,e.bw(this.b,63),this.d=A,x.Oy(this);},x.Vy=function(t){x.My();var A,n,P,r,i;for(e.bw(t,7),r=j(l.Feb,ht,6,t.g,16,1),P=0,n=0;n<t.g;n++)x.Zy(t,n)&&(r[n]=true,++P);for(i=j(l.j7,H,6,P,15,1),P=0,A=0;A<t.g;A++)r[A]&&(i[P++]=A);return i;},x.Wy=function(t,A,n,P){var r,i,D;for(r=0;r<2;r++)if(i=e.Zo(t,n[1],r),i!=n[0])return i==A||(D=e._o(t,n[1],r),n[0]=n[1],n[1]=i,t.j[i]==1)?false:(P!=null&&!e.Km(t,n[0])&&!e.Km(t,n[1])&&(P[0]=d.Math.max(P[0],D)),t.j[i]==2&&t.o[i]==2&&t.C[i]<=7?x.Wy(t,A,n,P):true);return  false;},x.Xy=function(t,A){var n;for(n=x.Ly[A]/2;t<-n;)t+=I1;for(;t>=n;)t-=x.Ly[A];return t;},x.Yy=function(t,A){return t.j[A]==2&&t.o[A]==2&&t.C[A]<=7;},x.Zy=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;if(e.fm(t,A)!=1||A<t.g&&e.pt(t.p,A)||(t.p&&A<t.g?e.ft(t.p,A):0)==3)return  false;for(n=j(l.j7,H,6,2,15,1),u=0;u<2;u++)if(n[u]=t.D[u][A],e.Km(t,n[u])||e.tp(t,n[u])<=1)return  false;if(x.Yy(t,n[0])||x.Yy(t,n[1])){for(s=j(l.j7,H,6,1,15,1),s[0]=A,P=j(l.j7,E1,7,2,0,2),o=0;o<2;o++)if(x.Yy(t,n[o])&&(P[o]=j(l.j7,H,6,2,15,1),P[o][0]=n[1-o],P[o][1]=n[o],!x.Wy(t,n[1-o],P[o],s)))return  false;if(A!=s[0])return  false;for(Q=0;Q<2;Q++)P[Q]!=null&&(n[Q]=P[Q][1]);}for(D=0;D<2;D++){if(i=e.$o(t,n[D]),i==1)return  false;for(h=0,L=0;L<i;L++)r=e.Zo(t,n[D],L),(t.u[r]&n0)!=0||++h;if(h<2)return  false;}return  true;},N(254,1,{},x.Ty,x.Uy),l.s8=O(254),x._y=function(){x._y=D1,x.$y=v(F(l.j7,2),E1,7,0,[v(F(l.j7,1),H,6,15,[0,0,1]),v(F(l.j7,1),H,6,15,[0,2,3]),v(F(l.j7,1),H,6,15,[1,3,3])]);},x.az=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It;if(e._v(t.c),t.d=null,e.bw(A,63),e.fm(A,n)!=1||n<A.g&&e.pt(A.p,n)||e.Tl(A,A.D[0][n])==1||e.Tl(A,A.D[1][n])==1||(Qt=(A.F[n]&128)!=0,Qt&&(A.p&&n<A.g?e.ft(A.p,n):0)<6))return  false;for(r=j(l.Feb,ht,6,A.f,16,1),t.a=0,I=0;I<2;I++){for(t.b[I]=A.D[I][n],t.e[I]=A.D[1-I][n];e.Qo(A,t.b[I])==2&&e.tp(A,t.b[I])==2&&e.Tl(A,t.b[I])<10;)for(ut=0;ut<e.$o(A,t.b[I]);ut++)if(D=e.Zo(A,t.b[I],ut),D!=t.e[I]&&A.C[D]!=1){if(A.j[D]==1||t.a==16)return  false;r[t.b[I]]=true,t.e[I]=t.b[I],t.b[I]=D,++t.a;break;}if(Dt=e.tp(A,t.b[I]),Dt>4||Dt==1)return  false;r[t.b[I]]=true;}for(Z=0;Z<2;Z++)for(ut=0;ut<e.$o(A,t.b[Z]);ut++)D=e.Zo(A,t.b[Z],ut),A.C[D]!=1&&(r[D]=true);for(t.g=j(l.j7,H,6,A.f,15,1),e.Co(A,t.c,r,true,t.g),z=0;z<t.g.length;z++)t.g[z]!=-1&&(t.i[t.g[z]]=z);if(e.On(t.c,true),Qt&&(i=e.Xo(t.c,t.g[t.b[0]],t.g[t.b[1]]),i!=-1)){for(e.Ln(t.c,i,256,true),Et=(e.bw(A,7),A.p),xt=0;xt<Et.j.a.length;xt++)if(e.rt(Et,xt,n)){for(tt=0;tt<2;tt++)for(ut=0;ut<e.$o(A,t.b[tt]);ut++)if(D=e.Zo(A,t.b[tt],ut),D!=t.e[tt]&&e.qt(Et,xt,D)&&A.C[D]!=1){e.Ln(t.c,e.Xo(t.c,t.g[t.b[tt]],t.g[D]),256,true);break;}}}for(ot=0;ot<2;ot++){for(e.$p(A,t.b[ot],true)&&e.vn(t.c,t.g[t.b[ot]],IA,true),o=false,ut=0;ut<e.$o(A,t.b[ot]);ut++)D=e.Zo(A,t.b[ot],ut),D!=t.e[ot]&&A.C[D]!=1&&(k=e.Xo(t.c,t.g[t.b[ot]],t.g[D]),e.im(t.c,k)==8?o=true:A.C[D]==6&&!e.Tp(A,t.b[ot])&&(B=D<A.f&&e.ot(A.p,D)?2:4,e.vn(t.c,t.g[D],B,true)),u=e._o(A,t.b[ot],ut),qt=A.p&&u<A.g?e.ft(A.p,u):0,(qt==3||qt==4)&&e.Ln(t.c,k,qt<<17,true),(u<A.g&&e.pt(A.p,u)||e.ap(A,t.b[ot],ut)==2)&&(Dt=e.tp(A,D),S=Dt==3,!S&&Dt==2&&(A.u[D]&V0)==0&&(S=e.Mp(A,t.b[1-ot],u)!=-1),S?e.vn(t.c,t.g[D],Re,true):u<A.g&&e.pt(A.p,u)&&e.vn(t.c,t.g[D],pe,true)),e.ap(A,t.b[ot],ut)==1&&(e.tp(A,D)==4?e.vn(t.c,t.g[D],le,true):A.C[D]==6&&e.vn(t.c,t.g[D],v0,true)));o||(e.Tp(A,t.b[ot])?e.vn(t.c,t.g[t.b[ot]],2,true):e.vn(t.c,t.g[t.b[ot]],4,true));}for(e.bw(t.c,191),V=0;V<2;V++)if(M=t.g[t.b[V]],e.Nl(t.c,M)==3){if(e.Nl(A,t.b[V])==3)return  false;lt=e.Ro(t.c,M),e.Mn(t.c,lt,257),e.Zl(t.c,0,lt)!=M&&(e.Dn(t.c,1,lt,e.Zl(t.c,0,lt)),e.Dn(t.c,0,lt,M)),e.bw(t.c,191);}if(h=t.g[t.b[0]],b=t.g[t.b[1]],wt=t.g[t.e[0]],bt=t.g[t.e[1]],Q=e.Hl(t.c,h),L=e.Hl(t.c,b),e.zm(t.c,h)&&e.zm(t.c,b)&&(Q!=0||L!=0)&&(Q!=L||e.Gl(t.c,h)!=e.Gl(t.c,b)))return  false;if(s=false,e.zm(t.c,h)&&Q!=0&&(e.on(t.c,h,0,-1),s=true),e.zm(t.c,b)&&L!=0&&(e.on(t.c,b,0,-1),s=true),s&&e.bw(t.c,191),ct=e.mw(t.c,h),yt=e.mw(t.c,b),ct<yt&&(It=t.b[0],t.b[0]=t.b[1],t.b[1]=It,It=t.e[0],t.e[0]=t.e[1],t.e[1]=It,It=h,h=b,b=It,It=wt,wt=bt,bt=It),it=false,(e.zm(t.c,h)||e.zm(t.c,b))&&(e.zm(t.c,h)?it=e.cw(t.c,h)==1:e.zm(t.c,b)&&(it=e.cw(t.c,b)==1),it)){for(P=0;P<t.c.q;P++)e.zn(t.c,P,-e.Ql(t.c,P));e.bw(t.c,191);}return c=x.cz(t,h,wt),g=x.cz(t,b,bt),t.f[0]=c==-1?-1:t.i[c],t.f[1]=g==-1?-1:t.i[g],G=e.jw(t.c),G==null?false:(p=x.bz(t,h,wt),q=x.bz(t,b,bt),p==0&&q==0&&(t.c.I&fA)==n0?Nt=2:Nt=x.$y[p][q],Zt=Nt==0?it?"<":">":Nt==1?it?"-":"+":Nt==3?"=":"",t.d=G+Zt,true);},x.bz=function(t,A,n){var P,r,i,D;if(e.$o(t.c,A)==2)return 1;if(P=x.dz(t,A,n),e.$o(t.c,A)==3)return e.mw(t.c,P[0])==e.mw(t.c,P[1])?e.Qo(t.c,A)==1&&e.Tl(t.c,A)<10||e.Tp(t.c,A)||e.$p(t.c,A,true)?2:1:e.Qo(t.c,A)==1&&e.Tl(t.c,A)<10||e.Tp(t.c,A)||e.$p(t.c,A,true)?1:0;if(e.$o(t.c,A)==4){for(r=0;r<P.length;r++)for(D=e.mw(t.c,P[r]),i=r+1;i<P.length;i++)if(D==e.mw(t.c,P[i]))return 1;}return 0;},x.cz=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h;for(L=-1,s=-1,h=-1,i=j(l.Feb,ht,6,e.$o(t.c,A),16,1),o=0;o<e.$o(t.c,A);o++)if(!i[o]&&(r=e.Zo(t.c,A,o),r!=n&&(D=e.mw(t.c,r),s<D))){for(u=false,Q=o+1;Q<e.$o(t.c,A);Q++)if(P=e.Zo(t.c,A,Q),P!=n&&e.mw(t.c,P)==D){if(i[Q]=true,u)return r;u=true;}u?h=r:(s=D,L=r);}return L==-1&&(e.Qo(t.c,A)==1&&e.Tl(t.c,A)<10||e.Tp(t.c,A)||e.$p(t.c,A,true))?h:L;},x.dz=function(t,A,n){var P,r,i;for(i=0,P=j(l.j7,H,6,e.$o(t.c,A)-1,15,1),r=0;r<e.$o(t.c,A);r++)e.Zo(t.c,A,r)!=n&&(P[i++]=e.Zo(t.c,A,r));return P;},x.ez=function(){x._y(),this.c=new e.uw(24,29),this.i=j(l.j7,H,6,24,15,1),this.b=j(l.j7,H,6,2,15,1),this.e=j(l.j7,H,6,2,15,1),this.f=j(l.j7,H,6,2,15,1);},N(250,1,{},x.ez),U.a=0,l.u8=O(250),x.fz=function(t){var A,n,P,r,i,D,u,o,Q;switch(A=null,t.e){case 1:for(A=j(l.Eeb,Yt,6,2*t.a.length,15,1),P=0;P<t.a.length;P++)A[P]=t.a[P],A[t.a.length+P]=180+t.a[P]<<16>>16;return A;case 2:for(r=t.a[0]==0?1:0,i=t.a[t.a.length-1]==180?t.a.length-1:t.a.length,Q=i-r,A=j(l.Eeb,Yt,6,t.a.length+Q,15,1),D=0;D<t.a.length;D++)A[D]=t.a[D];for(u=r;u<i;u++)A[A.length-1-u+r]=360-t.a[u]<<16>>16;return A;case 3:for(r=t.a[0]==0?1:0,i=t.a[t.a.length-1]==90?t.a.length-1:t.a.length,Q=i-r,A=j(l.Eeb,Yt,6,2*t.a.length+2*Q,15,1),o=0;o<t.a.length;o++)A[o]=t.a[o],A[t.a.length+Q+o]=180+t.a[o]<<16>>16;for(n=r;n<i;n++)A[t.a.length+Q-1-n+r]=180-t.a[n]<<16>>16,A[A.length-1-n+r]=360-t.a[n]<<16>>16;return A;default:return t.a;}},x.gz=function(t){var A,n,P,r,i,D,u,o,Q,L;switch(A=null,L=t.c.length,t.e){case 1:for(A=j(l.Eeb,Yt,6,2*L,15,1),P=0;P<L;P++)A[P]=t.c[P],A[L+P]=t.c[P];return A;case 2:for(r=t.a[0]==0?1:0,i=t.a[L-1]==180?L-1:L,Q=i-r,A=j(l.Eeb,Yt,6,L+Q,15,1),D=0;D<L;D++)A[D]=t.c[D];for(u=r;u<i;u++)A[A.length-1-u+r]=t.c[u];return A;case 3:for(r=t.a[0]==0?1:0,i=t.a[L-1]==90?L-1:L,Q=i-r,A=j(l.Eeb,Yt,6,2*L+2*Q,15,1),o=0;o<L;o++)A[o]=t.c[o],A[L+Q+o]=t.c[o];for(n=r;n<i;n++)A[L+Q-1-n+r]=t.c[n],A[A.length-1-n+r]=t.c[n];return A;default:return t.c;}},x.hz=function(t){var A,n,P,r,i,D,u,o,Q,L;switch(A=null,L=t.d.length,t.e){case 1:for(A=y1(l.Eeb,[P0,Yt],[16,6],15,[2*L,2],2),P=0;P<L;P++)A[P][0]=t.d[P][0],A[P][1]=t.d[P][1],A[L+P][0]=180+t.d[P][0]<<16>>16,A[L+P][1]=180+t.d[P][1]<<16>>16;return A;case 2:for(r=t.a[0]==0?1:0,i=t.a[L-1]==180?L-1:L,Q=i-r,A=y1(l.Eeb,[P0,Yt],[16,6],15,[L+Q,2],2),D=0;D<L;D++)A[D][0]=t.d[D][0],A[D][1]=t.d[D][1];for(u=r;u<i;u++)A[A.length-1-u+r][0]=360-t.d[u][1]<<16>>16,A[A.length-1-u+r][1]=360-t.d[u][0]<<16>>16;return A;case 3:for(r=t.a[0]==0?1:0,i=t.a[L-1]==90?L-1:L,Q=i-r,A=y1(l.Eeb,[P0,Yt],[16,6],15,[2*L+2*Q,2],2),o=0;o<L;o++)A[o][0]=t.d[o][0],A[o][1]=t.d[o][1],A[L+Q+o][0]=180+t.d[o][0]<<16>>16,A[L+Q+o][1]=180+t.d[o][1]<<16>>16;for(n=r;n<i;n++)A[L+Q-1-n+r][0]=180-t.d[n][1]<<16>>16,A[L+Q-1-n+r][1]=180-t.d[n][0]<<16>>16,A[A.length-1-n+r][0]=360-t.d[n][1]<<16>>16,A[A.length-1-n+r][1]=360-t.d[n][0]<<16>>16;return A;default:return t.d;}},x.iz=function(t,A,n,P,r,i){var D,u,o,Q,L,s;for(t.a[n]=A<<16>>16,t.c[n]=t.c[n]+t.c[P]<<16>>16,t.d[n][0]=r<<16>>16,t.d[n][1]=i<<16>>16,s=t.a.length-1,o=j(l.Eeb,Yt,6,s,15,1),Q=j(l.Eeb,Yt,6,s,15,1),L=j(l.Eeb,P0,16,s,0,2),u=0,D=0;D<t.a.length;D++)D!=P&&(o[u]=t.a[D],Q[u]=t.c[D],L[u]=t.d[D],++u);t.a=o,t.c=Q,t.d=L;},x.jz=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h;for(A=0;t.a.length!=0;){if(t.a.length==1){if(t.e==1||t.e==0)break;if(t.e==2&&(t.a[0]==0||t.a[0]==180))break;if(t.e==3&&(t.a[0]==0||t.a[0]==90))break;}for(h=t.a[0]!=0&&(t.e==2||t.e==3)?-1:0,s=-1,L=St,D=h;D<t.a.length;D++)o=D+1,n=D>=0?t.a[D]:-t.a[0],P=o<t.a.length?t.a[o]:t.e==1?t.a[0]+180:t.e==2?360-t.a[t.a[D]==180?D-1:D]:t.e==3?180-t.a[t.a[D]==90?D-1:D]:t.a[0]+360,L>P-n&&(L=P-n,s=D);if(L>15)break;i=s,u=i+1,i==-1?(A+=t.c[0],t.a[0]=0,t.c[0]=t.c[0]*2<<16>>16,t.d[0][0]=-t.d[0][1]):u<t.a.length?(Q=x.kz(t,i,u,t.a[u]),x.iz(t,Q,i,u,t.d[i][0],t.d[u][1])):t.e==1?(u=0,Q=x.kz(t,i,u,t.a[u]+180),Q<=180?x.iz(t,Q,i,u,t.d[i][0],t.d[u][1]+180):x.iz(t,Q-180,u,i,t.d[u][0],t.d[i][1]-180)):t.e==2?(u=t.a[i]==180?i-1:i,t.a[i]<180?(A+=t.c[i],t.a[i]=180,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=360-t.d[i][0]<<16>>16):(T.Zjb(),Q=x.kz(t,i,u,t.a[u]),x.iz(t,Q,u,i,t.d[u][0],t.d[i][1]))):t.e==3?(u=t.a[i]==90?i-1:i,t.a[i]<90?(A+=t.c[i],t.a[i]=90,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=180-t.d[i][0]<<16>>16):(T.Zjb(),Q=x.kz(t,i,u,t.a[u]),x.iz(t,Q,u,i,t.d[u][0],t.d[i][1]))):(u=0,Q=x.kz(t,i,u,t.a[u]+360),Q<=360?x.iz(t,Q,i,u,t.d[i][0],t.d[u][1]+360):x.iz(t,Q-360,u,i,t.d[u][0],t.d[i][1]-360));}if(A!=0)for(r=0;r<t.c.length;r++)t.c[r]=Rt(d.Math.round(100*t.c[r]/(100+A)))<<16>>16;},x.kz=function(t,A,n,P){return Rt(d.Math.round((t.a[A]*t.c[A]+P*t.c[n])/(t.c[A]+t.c[n])));},x.lz=function(t){this.e=t;},x.mz=function(t){var A;if(this.e=t.e,t.a!=null)for(this.a=j(l.Eeb,Yt,6,t.a.length,15,1),A=0;A<this.a.length;A++)this.a[A]=360-t.a[this.a.length-A-1]<<16>>16;if(t.d!=null)for(this.d=y1(l.Eeb,[P0,Yt],[16,6],15,[t.d.length,2],2),A=0;A<this.d.length;A++)this.d[A][0]=360-t.d[this.a.length-A-1][1]<<16>>16,this.d[A][1]=360-t.d[this.a.length-A-1][0]<<16>>16;if(t.c!=null)for(this.c=j(l.Eeb,Yt,6,t.c.length,15,1),A=0;A<this.c.length;A++)this.c[A]=t.c[this.c.length-A-1];if(t.b!=null)for(this.b=j(l.f7,Y,6,t.b.length,15,1),A=0;A<this.b.length;A++)this.b[A]=t.b[this.b.length-A-1];},N(194,1,{194:1},x.lz,x.mz),U.e=0,l.v8=O(194),x.nz=function(t,A){var n,P,r,i;n=A[1],P=A[2],r=t.j[n]-1,i=t.j[P]-1,r>3||i>3?(this.b=j(l.Eeb,Yt,6,4,15,1),this.b[0]=45,this.b[1]=135,this.b[2]=225,this.b[2]=315,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[4,2],2),this.c[0][0]=30,this.c[0][1]=60,this.c[1][0]=120,this.c[1][1]=150,this.c[2][0]=210,this.c[2][1]=240,this.c[2][0]=300,this.c[2][1]=330,this.a=j(l.Eeb,Yt,6,4,15,1),this.a[0]=25,this.a[1]=25,this.a[2]=25,this.a[3]=25):(t.o[n]==0||t.C[n]>9)&&(t.o[P]==0||t.C[P]>9)?r==3&&i==3||r==3&&i==2||r==3&&i==1||r==2&&i==3||r==1&&i==3||r==2&&i==2&&(A[0]!=-1||A[3]!=-1)?(this.b=j(l.Eeb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=j(l.Eeb,Yt,6,3,15,1),this.a[0]=33,this.a[1]=33,this.a[2]=33):r==1&&i==2&&A[3]==-1||r==2&&i==1&&A[0]==-1?(this.b=j(l.Eeb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=j(l.Eeb,Yt,6,3,15,1),this.a[0]=40,this.a[1]=20,this.a[2]=40):(r==1&&i==1||r==1&&i==2&&A[3]!=-1||r==2&&i==1&&A[0]!=-1||r==2&&i==2&&A[0]==-1&&A[3]==-1)&&(this.b=j(l.Eeb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=j(l.Eeb,Yt,6,3,15,1),this.a[0]=25,this.a[1]=50,this.a[2]=25):(t.o[n]==0||t.C[n]>9)&&t.o[P]==1||(t.o[P]==0||t.C[P]>9)&&t.o[n]==1?r==3||i==3?(this.b=j(l.Eeb,Yt,6,6,15,1),this.b[0]=0,this.b[1]=60,this.b[2]=120,this.b[3]=180,this.b[4]=240,this.b[5]=300,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=45,this.c[1][1]=75,this.c[2][0]=105,this.c[2][1]=135,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=225,this.c[4][1]=255,this.c[5][0]=285,this.c[5][1]=315,this.a=j(l.Eeb,Yt,6,6,15,1),this.a[0]=16,this.a[1]=16,this.a[2]=16,this.a[3]=16,this.a[4]=16,this.a[5]=16):r==1&&i==1?(this.b=j(l.Eeb,Yt,6,2,15,1),this.b[0]=120,this.b[1]=240,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=105,this.c[0][1]=135,this.c[1][0]=225,this.c[1][1]=255,this.a=j(l.Eeb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[n]==1&&r==2&&i==1||t.o[P]==1&&i==2&&r==1?(this.b=j(l.Eeb,Yt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=j(l.Eeb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[n]==1&&r==1&&i==2&&A[3]==-1||t.o[P]==1&&i==1&&r==2&&A[0]==-1?(this.b=j(l.Eeb,Yt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=j(l.Eeb,Yt,6,3,15,1),this.a[0]=60,this.a[1]=20,this.a[2]=20):t.o[n]==1&&r==1&&i==2&&A[3]!=-1||t.o[P]==1&&i==1&&r==2&&A[0]!=-1?(this.b=j(l.Eeb,Yt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=j(l.Eeb,Yt,6,3,15,1),this.a[0]=20,this.a[1]=40,this.a[2]=40):r==2&&i==2&&(A[0]==-1||A[3]==-1?(this.b=j(l.Eeb,Yt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=j(l.Eeb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):(this.b=j(l.Eeb,Yt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=j(l.Eeb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50)):t.o[n]==1&&t.o[P]==1?r==1&&i==1?(this.b=j(l.Eeb,Yt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=j(l.Eeb,Yt,6,2,15,1),this.a[0]=10,this.a[1]=90):(this.b=j(l.Eeb,Yt,6,6,15,1),this.b[0]=0,this.b[1]=50,this.b[2]=130,this.b[3]=180,this.b[4]=230,this.b[5]=310,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=35,this.c[1][1]=65,this.c[2][0]=115,this.c[2][1]=145,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=215,this.c[4][1]=245,this.c[5][0]=295,this.c[5][1]=325,this.a=j(l.Eeb,Yt,6,6,15,1),this.a[0]=40,this.a[1]=5,this.a[2]=5,this.a[3]=40,this.a[4]=5,this.a[5]=5):(this.b=j(l.Eeb,Yt,6,1,15,1),this.b[0]=180,this.c=y1(l.Eeb,[P0,Yt],[16,6],15,[1,2],2),this.c[0][0]=165,this.c[0][1]=195,this.a=j(l.Eeb,Yt,6,1,15,1),this.a[0]=100);},N(251,1,{},x.nz),l.w8=O(251),x.qz=function(){x.qz=D1,x.pz=v(F(l.i7,1),w0,6,15,[1,hn,Br,2.119999885559082,1.9800000190734863,1.909999966621399,1.7699999809265137,QP,1.5,dr,1.5800000429153442,2.5,2.509999990463257,2.25,2.190000057220459,mr,1.8899999856948853,hi,1.8300000429153442,2.7300000190734863,2.619999885559082,2.5799999237060547,Ti,bi,LP,LP,kr,ci,ci,2.380000114440918,2.390000104904175,Fr,vr,1.8799999952316284,hi,wi,2.25,3.2100000381469727,2.8399999141693115,2.75,2.5199999809265137,2.559999942779541,LP,kr,Ti,kr,2.1500000953674316,gi,ji,2.430000066757202,bi,Bi,1.9900000095367432,2.0399999618530273,2.059999942779541,3.4800000190734863,3.0299999713897705,2.9800000190734863,di,mi,2.950000047683716,mi,2.9000000953674316,ki,pr,Fi,ki,vi,pr,Fi,pi,2.740000009536743,2.630000114440918,gi,2.569999933242798,ji,2.4800000190734863,2.4100000858306885,vr,Fr,LP,Bi,2.5999999046325684,2.5399999618530273,2.5,2.5,2.5,2.5,2.5,pi,2.930000066757202,di,2.7100000381469727,2.819999933242798,vi,pr,Ri,3.4000000953674316,Ri,2.700000047683716]),x.oz=v(F(l.i7,1),w0,6,15,[.25,Rr,.46000000834465027,Ci,1.0199999809265137,Tn,.75,.7099999785423279,.6299999952316284,.6399999856948853,.9599999785423279,Mi,Hi,1.2599999904632568,Cr,qi,1.0299999713897705,.9900000095367432,Ei,Si,Vi,1.4800000190734863,Mr,xi,zi,1.190000057220459,Cr,qi,1.100000023841858,hn,hn,Ni,UP,UP,Cr,1.1399999856948853,UP,Ii,Hr,qr,1.5399999618530273,Oi,1.3799999952316284,Gi,1.25,1.25,hn,Hi,Ji,dr,yn,yn,Mr,Ci,1.350000023841858,Fr,Si,oP,qr,Zi,1.7400000095367432,Ki,Yi,yP,Xi,yP,_i,QP,Er,1.6399999856948853,bn,Wi,1.5199999809265137,dr,$i,1.309999942779541,ai,zi,1.2300000190734863,Ni,1.4199999570846558,1.5,Ji,1.5099999904632568,t9,Oi,t9,2.2300000190734863,2.009999990463257,wi,1.75,Xi,bn,Vi,Yi,QP,QP,yP,yP,Er,_i,Ki,Zi,1.6100000143051147,A9,1.4900000095367432,Br,1.409999966621399,xi,ai,Gi,UP,$i,Mr,Br,Wi,1.75,Er,A9]);},Xt.sz=function(){Xt.sz=D1,Xt.rz=e.po("X",32);},Xt.tz=function(t,A){Xt.sz();var n,P,r,i,D;for(r=0;r<t.q;r++)for(D=e.fw(t),e.Cn(D,r,Xt.rz),e.bw(D,15),P=0;P<D.f;P++)(D.u[P]&S0)!=0&&e.Ip(D,P)==-1&&(i=(e.bw(D,7),D.o[P]==2&&D.j[P]==2?e.nq(D,P,false):e.pq(D,P,false)),i!=-1&&(t.H[i]=257,t.T=0,t.D[1][i]==P&&(n=t.D[0][i],t.D[0][i]=P,t.T=0,t.D[1][i]=n,t.T=0),e.on(t,P,A,0)));},Xt.uz=function(t){Xt.sz();var A,n,P,r;for(Xt.tz(t,1),P=t.q,n=j(l.Ybb,st,2,P,6,1),A=0;A<P;A++)r=e.fw(t),e.mn(r,A,(e.Rk(),e.Lk)[r.C[A]]+"*"),r.C[A]==1?e.Cn(r,A,Xt.rz):e.tn(r,A,r.A[A]+5),Xt.vz(r),n[A]=e.rg(new e.Og(r,8));return n;},Xt.vz=function(t){var A;for(e.bw(t,15),A=0;A<t.q;A++)(t.u[A]&3)!=0&&e.on(t,A,1,0);},Xt.rz=0,Xt.wz=function(t,A,n){var P,r;for(P=y1(l.Ybb,[c0,st],[13,2],6,[t.f,A],2),e.bw(t,7),r=0;r<t.f;r++)P[r]=Xt.xz(t,r,A,n);return P;},Xt.xz=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g;for(o=new e.uw(t.f,t.g),L=new y.hxb(),b=0,h=0,D=j(l.Feb,ht,6,t.f,16,1),i=j(l.j7,H,6,t.f,15,1),g=0;g<n&&h<t.f;g++){if(h==0)i[0]=A,D[A]=true,h=1;else {for(c=h,Q=b;Q<h;Q++)for(r=i[Q],s=0;s<t.j[r];s++)if(u=t.i[r][s],!D[u])switch(P){case 0:D[u]=true,i[c++]=u;break;case 1:Xt.zz(t,r)&&Xt.zz(t,u)||(D[u]=true,i[c++]=u);}b=h,h=c;}e.Co(t,o,D,true,null),y.exb(L,e.rg(new e.Og(o,8)));}return y.gxb(L,j(l.Ybb,st,2,L.a.a.length,6,1));},Xt.yz=function(t,A,n){var P,r,i,D,u,o;for(u=e.Pq(new e._q(true),t),r=-1,i=0;i<u.q;i++)if(P=u.t==null||u.t[i]==null?null:T.ijb((D=u.t[i],S1.Mmb(),D)),P!=null&&(o=w.aAb("*").length,T.kjb(w.aAb(P).substr(w.aAb(P).length-o,o),"*"))){r=i;break;}return r>=0?Xt.xz(u,r,A,n):j(l.Ybb,st,2,0,6,1);},Xt.zz=function(t,A){return !(t.C[A]!=6||t.s[A]!=0||e.op(t,A)+t.j[A]!=4);},Xt.Kz=function(){Xt.Kz=D1,Xt.Az=d.Math.cos(Me),Xt.Fz=d.Math.sin(Me),Xt.Cz=d.Math.cos(Sr),Xt.Hz=d.Math.sin(Sr),Xt.Ez=d.Math.cos(b0),Xt.Jz=d.Math.sin(b0),Xt.Bz=d.Math.cos(cn),Xt.Gz=d.Math.sin(cn),Xt.Dz=d.Math.cos(Qr),Xt.Iz=d.Math.sin(Qr),d.Math.cos(e9),d.Math.sin(e9);},Xt.Lz=function(t){Xt.Kz();var A,n;for(e.bw(t,1),n=t.f,A=0;A<n;A++)Xt.Mz(t,A);},Xt.Mz=function(t,A){Xt.Kz();var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut;switch(I=e.op(t,A),Qt=t.J[A].a,ut=t.J[A].b,I){case 1:{if(z=t.j[A],z==0?(L=e.Vo(t,true),b=0):(L=Qt-e.Ql(t,t.i[A][0]),b=ut-e.Rl(t,t.i[A][0])),z==1)p=e.Tk(t,Qt+Xt.Dz*L+Xt.Iz*b,ut-Xt.Iz*L+Xt.Dz*b,0);else if(z==2)L=Qt-.5*(e.Ql(t,t.i[A][0])+e.Ql(t,t.i[A][1])),b=ut-.5*(e.Rl(t,t.i[A][0])+e.Rl(t,t.i[A][1])),p=e.Tk(t,Qt+L,ut+b,0);else if(z==3){for(ot=t.i[A][0],k=1;k<3;k++)u=t.k[A][k],(t.H[u]==129||t.H[u]==257)&&(ot=t.i[A][k]);if(n=d.Math.abs(e.oo(e.no(t.J[A].a,t.J[A].b,e.Ql(t,t.i[A][0]),e.Rl(t,t.i[A][0])),e.no(t.J[A].a,t.J[A].b,e.Ql(t,t.i[A][1]),e.Rl(t,t.i[A][1])))),P=d.Math.abs(e.oo(e.no(t.J[A].a,t.J[A].b,e.Ql(t,t.i[A][0]),e.Rl(t,t.i[A][0])),e.no(t.J[A].a,t.J[A].b,e.Ql(t,t.i[A][2]),e.Rl(t,t.i[A][2])))),r=d.Math.abs(e.oo(e.no(t.J[A].a,t.J[A].b,e.Ql(t,t.i[A][1]),e.Rl(t,t.i[A][1])),e.no(t.J[A].a,t.J[A].b,e.Ql(t,t.i[A][2]),e.Rl(t,t.i[A][2])))),tt=true,n>P&&n>r?P+r<A1&&(tt=false,L=Qt-.5*(e.Ql(t,t.i[A][0])+e.Ql(t,t.i[A][1])),b=ut-.5*(e.Rl(t,t.i[A][0])+e.Rl(t,t.i[A][1]))):P>n&&P>r?n+r<A1&&(tt=false,L=Qt-.5*(e.Ql(t,t.i[A][0])+e.Ql(t,t.i[A][2])),b=ut-.5*(e.Rl(t,t.i[A][0])+e.Rl(t,t.i[A][2]))):n+P<A1&&(tt=false,L=Qt-.5*(e.Ql(t,t.i[A][1])+e.Ql(t,t.i[A][2])),b=ut-.5*(e.Rl(t,t.i[A][1])+e.Rl(t,t.i[A][2]))),tt){for(G=t.i[A][0],Q=U0,B=0;B<3;B++)i=t.i[A][B],i!=ot&&(o=d.Math.pow(t.J[A].a-t.J[i].a,2)+d.Math.pow(t.J[A].b-t.J[i].b,2),o<Q&&(G=i,Q=o,T.Zjb()));p=e.Tk(t,(t.J[ot].a+t.J[G].a)/2,(t.J[ot].b+t.J[G].b)/2,0);}else p=e.Tk(t,Qt+L,ut+b,0);}else p=e.Tk(t,Qt+L,ut+b,0);e.Cn(t,p,1),e.Wk(t,A,p,1);}break;case 2:if(Z=t.j[A],Z==1)L=Qt-e.Ql(t,t.i[A][0]),b=ut-e.Rl(t,t.i[A][0]),p=e.Tk(t,Qt+(Xt.Ez*L-Xt.Jz*b)*.7,ut+(Xt.Jz*L+Xt.Ez*b)*.7,0),e.Cn(t,p,1),e.Wk(t,A,p,1),p=e.Tk(t,Qt+(Xt.Bz*L-Xt.Gz*b)*.7,ut+(Xt.Gz*L+Xt.Bz*b)*.7,0),e.Cn(t,p,1),e.Wk(t,A,p,1);else if(Z==2)s=Qt-e.Ql(t,t.i[A][0]),c=ut-e.Rl(t,t.i[A][0]),h=Qt-e.Ql(t,t.i[A][1]),g=ut-e.Rl(t,t.i[A][1]),S=d.Math.sqrt(s*s+c*c)*.7,V=d.Math.sqrt(h*h+g*g)*.7,L=s+h,b=c+g,q=d.Math.sqrt(L*L+b*b),D=(S+V)/2,L=L/q*D,b=b/q*D,it=e.Ip(t,A),p=e.Tk(t,Qt+Xt.Az*L-Xt.Fz*b,ut+Xt.Fz*L+Xt.Az*b,0),e.Cn(t,p,1),it>-1?e.Wk(t,A,p,1):e.Wk(t,A,p,257),p=e.Tk(t,Qt+Xt.Cz*L-Xt.Hz*b,ut+Xt.Hz*L+Xt.Cz*b,0),e.Cn(t,p,1),e.Wk(t,A,p,1);else for(M=0;M<I;M++)p=e.Tk(t,Qt,ut,0),e.Cn(t,p,1),e.Wk(t,A,p,1);break;case 3:t.i[A][0]>0?(L=(Qt-e.Ql(t,t.i[A][0]))*.7,b=(ut-e.Rl(t,t.i[A][0]))*.7,p=e.Tk(t,Qt+L,ut+b,0),e.Cn(t,p,1),e.Wk(t,A,p,1),p=e.Tk(t,Qt-b,ut+L,0),e.Cn(t,p,1),e.Wk(t,A,p,1),p=e.Tk(t,Qt+b,ut-L,0),e.Cn(t,p,1),e.Wk(t,A,p,1)):(L=e.Vo(t,true),b=e.Vo(t,true),p=e.Tk(t,Qt+L,ut+b,0),e.Cn(t,p,1),e.Wk(t,A,p,1),p=e.Tk(t,Qt-b*Xt.Ez,ut+L*Xt.Jz,0),e.Cn(t,p,1),e.Wk(t,A,p,1),p=e.Tk(t,Qt-b*Xt.Ez,ut-L*Xt.Jz,0),e.Cn(t,p,1),e.Wk(t,A,p,1));break;default:{for(M=0;M<I;M++)p=e.Tk(t,Qt,ut,0),e.Cn(t,p,1),e.Wk(t,A,p,1);break;}}},Xt.Nz=function(t){Xt.Kz();var A,n;for(e.bw(t,1),n=0,A=0;A<t.q;A++)t.C[A]==1?++n:n+=t.e[A]-t.j[A]+e.op(t,A);return n;},Xt.Az=0,Xt.Bz=0,Xt.Cz=0,Xt.Dz=0,Xt.Ez=0,Xt.Fz=0,Xt.Gz=0,Xt.Hz=0,Xt.Iz=0,Xt.Jz=0,X.Pz=function(t,A,n){var P,r,i;for(i=A.length,P=new X.uB(t.f,i,false),P.a[0]=0,P.b[0]=0,r=0;r<i;r++)P.k[r]=128-i,P.e[r]=A[r];i<8?X.Yz(P):X.Wz(t,P,A,n),X.FA(t.d,P);},X.Qz=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h;for(D=j(l.j7,H,6,r,15,1),i=0;i<t.f.q;i++){for(o=0;o<e.Lo(t.f,i);o++){for(h=n[e.Zo(t.f,i,o)],Q=0;Q<o&&!(h<D[Q]);Q++);for(L=o;L>Q;L--)D[L]=D[L-1];D[Q]=h;}for(s=e.Lo(t.f,i),e.eh(A[i],i),e.ah(A[i],P,l.Ueb(n[i])),e.ah(A[i],(r-s)*(P+1),0),u=0;u<s;u++)e.ah(A[i],P+1,l.Ueb(D[u]));}},X.Rz=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(P=e.Pg(t.f.f),o=2,n=0;n<t.f.f;n++)o=d.Math.max(o,e.Lo(t.f,n));for(i=(62+2*P+o*(P+1))/63|0,r=j(l.v7,He,101,t.f.q,0,1),A=0;A<t.f.q;A++)r[A]=new e.fh(i),e.eh(r[A],A);for(s=j(l.j7,H,6,t.f.q,15,1),D=0;D<t.f.g;D++)u=e.gm(t.f,D),(u==1||u==2)&&(e.bh(r[e.Zl(t.f,0,D)],l.Ueb(u)),e.bh(r[e.Zl(t.f,1,D)],l.Ueb(u)));Q=X.Sz(r,s);do L=Q,X.Qz(t,r,s,P,o),Q=X.Sz(r,s);while(L!=Q);return s;},X.Sz=function(t,A){var n,P;for(P=0,y._qb(t,0,t.length,null),n=0;n<t.length;n++)(n==0||e.dh(t[n],t[n-1])!=0)&&++P,A[t[n].a]=P;return P;},X.Tz=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h;for(o=new y.Bqb(t.d);o.a<o.c.a.length;)for(u=y.Aqb(o),Q=0;Q<u.f.length;Q++)if(P=u.f[Q],e.fm(t.f,P)==2&&(!e.jq(t.f,P)&&(e.gm(t.f,P)==3||e.gm(t.f,P)==0)&&e.Kn(t.f,P),!e.gq(t.f,P)&&e.$o(t.f,e.Zl(t.f,0,P))>1&&e.$o(t.f,e.Zl(t.f,1,P))>1&&(e.gm(t.f,P)==1||e.gm(t.f,P)==2))){for(h=j(l.j7,H,6,2,15,1),r=j(l.j7,H,6,2,15,1),L=0;L<2;L++)for(h[L]=t.f.M,r[L]=e.Zl(t.f,L,P),s=0;s<e.Lo(t.f,r[L]);s++)i=e.Zo(t.f,r[L],s),i!=e.Zl(t.f,1-L,P)&&h[L]>i&&(h[L]=i);D=X.$A(u.a[u.g[r[0]]],u.b[u.g[r[0]]],u.a[u.g[r[1]]],u.b[u.g[r[1]]]),A=X.$A(u.a[u.g[h[0]]],u.b[u.g[h[0]]],u.a[u.g[r[0]]],u.b[u.g[r[0]]]),n=X.$A(u.a[u.g[r[1]]],u.b[u.g[r[1]]],u.a[u.g[h[1]]],u.b[u.g[h[1]]]),X.$z(D,A)<0^X.$z(D,n)<0^e.gm(t.f,P)==2&&X.iB(u,P);}},X.Uz=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;for(h=new y.Mj(),k=new y.Mj(),P=new y.Mj(),i=new y.Bqb(t.d);i.a<i.c.a.length;){for(r=y.Aqb(i),D=0,w.uzb(P.a,0),u=0;u<r.e.length;u++)A=r.e[u],n=t.i[A],n!=0&&(y.sj(P,new X.aB(r,u,n)),D+=n);if(D!=0)for(y.Jj(P,new X.RA()),Q=new y.Bqb(P);Q.a<Q.c.a.length&&(o=y.Aqb(Q),!(D*o.b>0&&(n=d.Math.abs(D)>=d.Math.abs(o.b)?o.b:D,D-=n,y.sj(n<0?h:k,new X.aB(r,o.a,n)),D==0))););}if(h.a.length==0||k.a.length==0)return null;for(y.Jj(k,new X.KA()),y.Jj(h,new X.OA()),g=new y.Bqb(k);g.a<g.c.a.length;)for(b=y.Aqb(g),s=new y.Bqb(h);s.a<s.c.a.length;)if(L=y.Aqb(s),b.b==-L.b)return t.i[X.mB(b.c,b.a)]-=b.b,t.i[X.mB(L.c,L.a)]-=L.b,new X.XA(b.c,L.c,b.a,L.a);for(B=new y.Bqb(k);B.a<B.c.a.length;)for(b=y.Aqb(B),s=new y.Bqb(h);s.a<s.c.a.length;)if(L=y.Aqb(s),b.b>-L.b)return t.i[X.mB(b.c,b.a)]+=L.b,t.i[X.mB(L.c,L.a)]-=L.b,new X.XA(b.c,L.c,b.a,L.a);for(c=new y.Bqb(k);c.a<c.c.a.length;)for(b=y.Aqb(c),s=new y.Bqb(h);s.a<s.c.a.length;)if(L=y.Aqb(s),b.b<-L.b)return t.i[X.mB(b.c,b.a)]-=b.b,t.i[X.mB(L.c,L.a)]+=b.b,new X.XA(b.c,L.c,b.a,L.a);return null;},X.Vz=function(t){return t.d.a.length<2?null:new X.WA(y.xj(t.d,0),y.xj(t.d,1));},X.Wz=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z;if(L=v(F(l.h7,2),i1,15,0,[v(F(l.h7,1),Ft,6,15,[20]),null,null,v(F(l.h7,1),Ft,6,15,[0,10]),null,null,v(F(l.h7,1),Ft,6,15,[-4,12]),v(F(l.h7,1),Ft,6,15,[0,0,-7.5]),null,null,null,null,v(F(l.h7,1),Ft,6,15,[8.571428571428571,-8.571428571428571]),null,null,null,v(F(l.h7,1),Ft,6,15,[-2.4])]),s=v(F(l.j7,2),E1,7,0,[v(F(l.j7,1),H,6,15,[146]),v(F(l.j7,1),H,6,15,[627]),null,v(F(l.j7,1),H,6,15,[2457,1170]),null,v(F(l.j7,1),H,6,15,[2451,8643,2519]),v(F(l.j7,1),H,6,15,[9362,14798]),v(F(l.j7,1),H,6,15,[34377,-2147448999,26214]),null,v(F(l.j7,1),H,6,15,[37449,137313,95703,34371,37815,54891,132867,-2147309741,54857,55129,-2147449005,-2147449065]),null,v(F(l.j7,1),H,6,15,[530697,531819,899169,137289,694617,-2146951863,-2146952797,-2146939175,-2146929547,-2146929564,-2146625111,-2146931799,-2146940503,-2146931935]),v(F(l.j7,1),H,6,15,[1007293,610915]),v(F(l.j7,1),H,6,15,[542985,137283,2122017,530691,2206773,-2144711351,219209,2840841,137555,-2146871031,-2147264167,613705,-2145360543,-2146625271,694611,2454837,-2145356703,-2147345133,-2146928951,-2146931805,-2144641719,-2146951869,-2146625237,-2146624183,2841963,1074905,-2146625117,2799955,-2144723645,138583,859225,-2145264843,-2145216253,-2146624149,-2144700727,-2146928917,-2143905527,-2144045771,-2146789097,2288547,544407,2104323,-2146911977,-2144479405,3633737,-2146870089,-2146952169]),null,v(F(l.j7,1),H,6,15,[8487297,2172633,2116611,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8829813]),v(F(l.j7,1),H,6,15,[14071213])]),M=1<<A.e.length,D=0,o=0,A.e.length>7)for(b=0;b<A.e.length;b++)u=X.cA(t,n,P,b),u==1?D+=M:u==2&&(o+=M),D>>>=1,o>>>=1;if(S=A.e.length-9,A.e.length>=9&&A.e.length<=25&&s[S]!=null)for(z=0;z<s[S].length;z++)for(k=(P1&s[S][z])==0,Q=St&s[S][z],g=false;!g;g=!g){if(g){if(k)break;for(p=0,r=1;r!=M;r<<=1)p<<=1,(Q&r)!=0&&(p|=1);Q=p;}for(V=0;V<A.e.length;V++){if((Q&D)==0&&(~Q&o)==0){for(i=0,h=n9*(L[S]==null?0:L[S][z]),q=0,I=Q,B=true,c=0;c<A.e.length;c++)B&&++q,(I&1)==0&&(B=!B),I>>>=1;for(Z=q>(A.e.length/2|0),b=1;b<A.e.length;b++)A.a[b]=A.a[b-1]+d.Math.sin(i),A.b[b]=A.b[b-1]+d.Math.cos(i),(Q&1)==0&&(Z=!Z),i+=h+(Z?b0:cn),Q>>>=1;return;}(Q&1)!=0&&(Q|=M),Q>>>=1;}}X.Zz(A,D,o);},X.Xz=function(t){var A,n,P,r,i,D,u,o,Q,L;for(A=null,Q=null,D=0;D<t.f.g;D++)if(e.im(t.f,D)==16){for(n=e.Zl(t.f,0,D),r=-1,u=0;u<t.d.a.length&&(r=X.nB(y.xj(t.d,u),n),r==-1);u++);for(P=e.Zl(t.f,1,D),i=-1,o=0;o<t.d.a.length&&(i=X.nB(y.xj(t.d,o),P),i==-1);o++);u!=o&&(u>o&&(L=u,u=o,o=L,L=r,r=i,i=L),Q==null&&(Q=j(l.C8,c0,450,t.d.a.length,0,2)),Q[o]==null&&(Q[o]=j(l.C8,{450:1,4:1,5:1,8:1},106,o,0,1)),Q[o][u]?X.UA(Q[o][u],r,i):($1(Q[o],u,new X.XA(y.xj(t.d,u),y.xj(t.d,o),r,i)),!A&&(A=new y.Mj()),y.sj(A,Q[o][u])));}return A;},X.Yz=function(t){var A,n;for(A=A1-A1*(t.e.length-2)/t.e.length,n=1;n<t.e.length;n++)t.a[n]=t.a[n-1]+d.Math.sin(A*(n-1)),t.b[n]=t.b[n-1]+d.Math.cos(A*(n-1));},X.Zz=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;if(A==0||(A&n)!=0){X.Yz(t);return;}for(B=-1,k=0,i=1<<t.e.length-2,r=1<<t.e.length-1,u=1,D=2,Q=0;Q<t.e.length;Q++)(n&(r|u))==0&&(A&(r|u))!=0&&(A&i)==0&&(c=0,(n&i)!=0&&(c+=4),(A&r)!=0&&(c+=2),(A&u)!=0&&(c+=1),k<c&&(k=c,B=Q)),i=r,r=u,u=D,D=1<<(Q+2<t.e.length?Q+2:Q+2-t.e.length);if(B==-1){X.Yz(t);return;}for(h=0,h|=1<<B,b=2;b<t.e.length-1;){if(s=B+b<t.e.length?B+b:B+b-t.e.length,r=1<<(s==0?t.e.length-1:s-1),(n&r)!=0){++b;continue;}if(u=1<<s,(A&r)!=0){if((n&u)!=0){X.Yz(t);return;}h|=u,b+=2;continue;}if(D=1<<(s+1<t.e.length?s+1:s+1-t.e.length),(A&u)!=0&&(n&D)!=0){h|=u,b+=3;continue;}++b;}if(h==0){X.Yz(t);return;}for(P=A1-A1*(t.e.length-2)/t.e.length,L=1;L<t.e.length;L++)t.a[L]=t.a[L-1]+d.Math.sin(P*(L-1)),t.b[L]=t.b[L-1]+d.Math.cos(P*(L-1));for(u=1,g=2*d.Math.sin(P/2),o=0;o<t.e.length;o++)(h&u)!=0&&(t.a[o]+=g*d.Math.cos(P*(o-.5)),t.b[o]-=g*d.Math.sin(P*(o-.5))),u<<=1;},X.$z=function(t,A){var n;for(n=t-A;n<lr;)n+=I1;for(;n>A1;)n-=I1;return n;},X._z=function(t,A,n){var P,r;for(P=0,r=0;r<e.Lo(t.f,n);r++)X.pB(A,e.Zo(t.f,n,r))&&++P;return P;},X.aA=function(t,A,n,P){var r,i,D,u,o;return u=X.nB(A,P),o=X.nB(n,P),X.tB(n,A.a[u]-n.a[o],A.b[u]-n.b[o]),r=X.zA(t,A,P),i=X.zA(t,n,P),D=0,X._z(t,A,P)==1&&X._z(t,n,P)==1&&(D=b0),X.sB(n,n.a[o],n.b[o],r-i+D+A1),X.fA(t,A,n,1);},X.bA=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt;for(p=j(l.j7,H,6,r,15,1),q=j(l.j7,H,6,r,15,1),g=0;g<r;g++)p[g]=X.nB(A,P[g]),q[g]=X.nB(n,P[g]);for(ot=0,it=0,G=0,Qt=0,B=0;B<r;B++)ot+=A.a[p[B]],it+=A.b[p[B]],G+=n.a[q[B]],Qt+=n.b[q[B]];for(ot/=r,it/=r,G/=r,Qt/=r,X.tB(n,ot-G,it-Qt),Q=j(l.D8,C1,41,r,0,1),s=j(l.D8,C1,41,r,0,1),i=j(l.D8,C1,41,r,0,1),D=j(l.D8,C1,41,r,0,1),k=0;k<r;k++)Q[k]=new X.ZA(ot,it,A.a[p[k]],A.b[p[k]]),s[k]=new X.ZA(ot,it,n.a[q[k]],n.b[q[k]]),i[k]=new X.YA(Q[k].a-s[k].a,Q[k].b*s[k].b),D[k]=new X.YA(Q[k].a+s[k].a,Q[k].b*s[k].b);for(V=X.DA(i,r),I=X.DA(D,r),Dt=0,lt=0,M=0;M<r;M++)for(S=0;S<e.Lo(t.f,P[M]);S++)u=e.Zo(t.f,P[M],S),X.pB(A,u)&&!X.pB(n,u)&&++Dt,!X.pB(A,u)&&X.pB(n,u)&&++lt;for(L=j(l.D8,C1,41,Dt,0,1),h=j(l.D8,C1,41,lt,0,1),b=j(l.D8,C1,41,lt,0,1),Dt=0,lt=0,c=0;c<r;c++)for(S=0;S<e.Lo(t.f,P[c]);S++)u=e.Zo(t.f,P[c],S),X.pB(A,u)&&!X.pB(n,u)&&(o=X.nB(A,u),L[Dt]=new X.ZA(A.a[p[c]],A.b[p[c]],A.a[o],A.b[o]),++Dt),!X.pB(A,u)&&X.pB(n,u)&&(o=X.nB(n,u),ut=new X.ZA(n.a[q[c]],n.b[q[c]],n.a[o],n.b[o]),h[lt]=new X.YA(V.a+ut.a,ut.b),b[lt]=new X.YA(I.a-ut.a,ut.b),++lt);return Z=X.DA(L,Dt),z=X.DA(h,lt),tt=X.DA(b,lt),d.Math.abs(X.$z(Z.a,z.a))>d.Math.abs(X.$z(Z.a,tt.a))?X.sB(n,ot,it,V.a):(X.hB(n,ot,it),X.sB(n,ot,it,I.a)),X.fA(t,A,n,r);},X.cA=function(t,A,n,P){var r,i,D,u,o,Q;return i=P==A.length-1?0:P+1,u=P==0?A.length-1:P-1,D=i==A.length-1?0:i+1,e.fm(t.f,n[P])==2&&(r=e.gm(t.f,n[P]),r==1||r==2)?(X.kA(t,A[u],A[P],A[i])^X.kA(t,A[D],A[i],A[P])&&(r=r==1?2:1),r):e.jq(t.f,n[P])?(o=e.lt(e.Ep(t.f),n[u],n[P]),Q=e.lt(e.Ep(t.f),n[i],n[P]),o!=-1||Q!=-1?o==Q?2:1:2):0;},X.dA=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b;for(r=j(l.j7,H,6,t.f.q,15,1),i=j(l.j7,H,6,t.f.q,15,1),D=j(l.j7,H,6,t.f.q,15,1),u=j(l.j7,H,6,t.f.q,15,1),r[0]=A,D[A]=1,u[0]=-1,P=0,o=0;P<=o;){if(P==0||!t.b[r[P]])for(Q=0;Q<e.Lo(t.f,r[P]);Q++)n=e.Zo(t.f,r[P],Q),h=e._o(t.f,r[P],Q),D[n]==0&&!t.c[h]&&(r[++o]=n,i[o]=h,D[n]=D[r[P]]+1,u[o]=P);if(P==o){for(b=new X._A(D[r[P]]),L=P,s=0;s<b.a.length;s++)b.a[s]=r[L],b.b[s]=i[L],L=u[L];return b;}++P;}return null;},X.eA=function(t){var A,n,P,r;for(r=0,P=null,n=new y.Bqb(t);n.a<n.c.a.length;)A=y.Aqb(n),r<A.b[0].e.length*A.b[1].e.length&&(r=A.b[0].e.length*A.b[1].e.length,P=A);return P;},X.fA=function(t,A,n,P){var r,i,D,u,o;for(i=new X.uB(t.f,A.e.length+n.e.length-P,A.i|n.i),r=0,u=0;u<A.e.length;u++)i.e[r]=A.e[u],i.k[r]=A.k[u],i.a[r]=A.a[u],i.b[r++]=A.b[u];for(D=0;D<n.e.length;D++)o=X.nB(A,n.e[D]),o==-1?(i.e[r]=n.e[D],i.k[r]=n.k[D],i.a[r]=n.a[D],i.b[r++]=n.b[D]):i.k[o]<n.k[D]&&(i.k[o]=n.k[D],i.a[o]=n.a[D],i.b[o]=n.b[D]);return i;},X.gA=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;for(u=j(l.j7,H,6,t.f.q,15,1),o=j(l.j7,H,6,t.f.q,15,1),Q=j(l.j7,H,6,t.f.q,15,1),L=j(l.j7,H,6,t.f.q,15,1),u[0]=n,Q[n]=1,L[0]=-1,D=0,s=0;D<=s;){for(h=0;h<e.Lo(t.f,u[D]);h++){if(r=e.Zo(t.f,u[D],h),c=e._o(t.f,u[D],h),r==A){for(i=Q[u[D]],P=j(l.j7,H,6,i,15,1),P[0]=c,b=1;b<i;b++)P[b]=o[D],D=L[D];return P;}Q[r]==0&&(u[++s]=r,o[s]=c,Q[r]=Q[u[D]]+1,L[s]=D);}if(D==s)return null;++D;}return null;},X.hA=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c;for(n=e.Zl(t.f,0,A),P=e.Zl(t.f,1,A),D=j(l.j7,H,6,t.f.q,15,1),u=j(l.j7,H,6,t.f.q,15,1),o=j(l.j7,H,6,t.f.q,15,1),Q=j(l.j7,H,6,t.f.q,15,1),D[0]=n,D[1]=P,u[1]=A,o[n]=1,o[P]=2,Q[0]=-1,Q[1]=0,i=1,L=1;i<=L;){for(s=0;s<e.$o(t.f,D[i]);s++){if(r=e.Zo(t.f,D[i],s),i>1&&r==n){for(c=new X._A(o[D[i]]),u[0]=e._o(t.f,D[i],s),h=i,b=0;b<c.a.length;b++)c.a[b]=D[h],c.b[b]=u[h],h=Q[h];return c;}o[r]==0&&e.fq(t.f,r)&&(D[++L]=r,u[L]=e._o(t.f,D[i],s),o[r]=o[D[i]]+1,Q[L]=i);}++i;}return null;},X.iA=function(t,A,n,P){var r,i,D,u,o,Q;for(D=j(l.j7,H,6,t.f.q,15,1),u=j(l.j7,H,6,t.f.q,15,1),D[0]=n,D[1]=A,u[n]=1,u[A]=2,i=1,o=1;i<=o;){for(Q=0;Q<e.$o(t.f,D[i]);Q++){if(r=e.Zo(t.f,D[i],Q),r==P)return 1+u[D[i]];u[r]==0&&e.fq(t.f,r)&&(D[++o]=r,u[r]=u[D[i]]+1);}++i;}return 0;},X.jA=function(t,A){var n,P,r,i,D,u,o;for(u=(A.T&15)!=0,o=A.T&248,!t.g&&(t.g=new y.Dvb()),(t.e&2)!=0&&e.qq(A,false),t.f=A,e.bw(t.f,7),t.d=new X.HA(),t.b=j(l.Feb,ht,6,t.f.q,16,1),t.c=j(l.Feb,ht,6,t.f.r,16,1),t.i=j(l.j7,H,6,t.f.q,15,1),n=0;n<t.f.q;n++)t.i[n]=e.Bl(t.f,n);for((t.e&12)!=0&&X.uA(t),(t.e&1)==0&&X.Oz&&X.wA(t,X.Oz),X.tA(t),X.pA(t),X.rA(t),X.pA(t),i=new y.Bqb(t.d);i.a<i.c.a.length;)P=y.Aqb(i),X.qB(P);for(X.Tz(t),X.xA(t),X.vA(t),X.oA(t),X.mA(t),X.qA(t),r=new y.Bqb(t.d);r.a<r.c.a.length;)for(P=y.Aqb(r),D=0;D<P.e.length;D++)e.zn(t.f,P.e[D],P.a[D]),e.An(t.f,P.e[D],P.b[D]),e.Bn(t.f,P.e[D],0);u&&(e.vq(t.f,o),e.yq(t.f)),t.a&&e.Zm(t.f);},X.kA=function(t,A,n,P){var r,i;for(i=0;i<e.$o(t.f,n);i++)if(r=e.Zo(t.f,n,i),r!=P&&r<A)return  false;return  true;},X.lA=function(t,A,n){var P;X.VA(A,n,(t.e&12)!=0),P=X.fA(t,A.b[0],A.b[1],0),X.AA(t,A.b[0],A.b[1],P);},X.mA=function(t){var A;for(A=X.Uz(t);A;)X.lA(t,A,yn),A=X.Uz(t);},X.nA=function(t,A,n,P){var r,i,D,u,o;for(r=j(l.j7,H,6,P,15,1),i=0,D=0;D<A.e.length;D++)for(u=0;u<n.e.length;u++)A.e[D]==n.e[u]&&(r[i++]=A.e[D]);o=P==1?X.aA(t,A,n,r[0]):X.bA(t,A,n,r,P),X.AA(t,A,n,o);},X.oA=function(t){var A,n;for(n=X.Xz(t);n;)A=X.eA(n),X.lA(t,A,hn),n=X.Xz(t);},X.pA=function(t){for(var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;;){for(M=0,b=0,B=null,k=null,D=1;D<t.d.a.length;D++)for(P=y.xj(t.d,D),u=0;u<D;u++){for(r=y.xj(t.d,u),A=0,n=0,c=0,g=0,L=0;L<P.e.length;L++)for(h=0;h<r.e.length;h++)P.e[L]==r.e[h]&&(++n,A=P.e[L],c<P.k[L]&&(c=P.k[L]),g<r.k[h]&&(g=r.k[h]));if(n>0&&(i=n==1&&X._z(t,P,A)==1&&X._z(t,r,A)==1?0:1,c>g?o=(i<<24)+(c<<16)+(g<<8)+n:o=(i<<24)+(g<<16)+(c<<8)+n,M<o)){for(M=o,b=n,c=0,g=0,s=0;s<P.e.length;s++)c<P.k[s]&&(c=P.k[s]);for(Q=0;Q<r.e.length;Q++)g<r.k[Q]&&(g=r.k[Q]);c>g?(B=P,k=r):(B=r,k=P);}}if(M==0)break;b==B.e.length?y.Fj(t.d,B):b==k.e.length?y.Fj(t.d,k):X.nA(t,B,k,b);}},X.qA=function(t){var A;for(A=X.Vz(t);A;)X.lA(t,A,Mi),A=X.Vz(t);},X.rA=function(t){for(var A,n,P,r,i,D,u;;){for(i=null,A=0;A<t.f.q;A++){for(u=0,r=0;r<e.Lo(t.f,A);r++)t.c[e._o(t.f,A,r)]||++u;u==1&&(D=X.dA(t,A),(!i||D.a.length>i.a.length)&&(i=D));}if(!i)break;for(n=new X.uB(t.f,i.a.length,false),P=0;P<i.a.length;P++)t.b[i.a[P]]=true,P<i.a.length-1&&(t.c[i.b[P]]=true),n.e[P]=i.a[P],n.a[P]=d.Math.cos(Me)*P,n.b[P]=(P&1)==1?0:.5,n.k[P]=128+i.a.length;X.FA(t.d,n);}},X.sA=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h;for(i=0;i<t.f.r;i++)if(P=e.Zl(t.f,0,i),r=e.Zl(t.f,1,i),!(e.gq(t.f,i)||e.fm(t.f,i)!=1||e.Lo(t.f,P)==1||e.Lo(t.f,r)==1)&&!((t.e&4)!=0&&e.Km(t.f,P)&&e.Km(t.f,r))){for(s=false,Q=0;Q<2;Q++)if(D=e.Zl(t.f,Q,i),e.Lo(t.f,D)>2){for(h=true,o=-1,L=0;L<e.Lo(t.f,D);L++)u=e.Zo(t.f,D,L),u!=e.Zl(t.f,1-Q,i)&&(o==-1?o=n[u]:o!=n[u]&&(h=false));if(h){s=true;break;}}s||((t.e&8)!=0&&e.Km(t.f,P)&&e.Km(t.f,r)?A[i]=1:A[i]=2);}},X.tA=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct;for(P=0;P<t.f.f;P++)if(e.Lo(t.f,P)>4){for(h=new X.uB(t.f,1+e.Lo(t.f,P),false),h.a[e.Lo(t.f,P)]=0,h.b[e.Lo(t.f,P)]=0,h.k[e.Lo(t.f,P)]=32,h.e[e.Lo(t.f,P)]=P,t.b[P]=true,c=0;c<e.Lo(t.f,P);c++)Q=e.Zo(t.f,P,c),h.a[c]=d.Math.sin(b0*c-Ve),h.b[c]=d.Math.cos(b0*c-Ve),h.k[c]=32,h.e[c]=Q,t.b[Q]=true,t.c[e._o(t.f,P,c)]=true;X.FA(t.d,h);}for(Qt=e.Ep(t.f),it=0;it<Qt.j.a.length;it++){if(ut=y.xj(Qt.k,it).length,ot=y.xj(Qt.j,it),Dt=false,(t.e&12)!=0){for(Dt=true,c=0;c<ut;c++)if(!e.Km(t.f,ot[c])){Dt=false;break;}}if(!Dt){for(k=false,g=0;g<ut;g++)if(e.Uo(t.f,ot[g])==ut){k=true;break;}if(k)for(G=y.xj(Qt.k,it),X.Pz(t,ot,G),c=0;c<ut;c++)t.b[ot[c]]=true,t.c[G[c]]=true;}}for(u=0;u<t.f.g;u++)if(e.gq(t.f,u)&&!t.c[u])for(ct=X.hA(t,u),ot=ct.a,G=ct.b,X.Pz(t,ot,G),c=0;c<ct.a.length;c++)t.b[ot[c]]=true,t.c[G[c]]=true;for(o=0;o<t.f.r;o++)if(!t.c[o]&&e.fm(t.f,o)==3&&(r=e.Zl(t.f,0,o),i=e.Zl(t.f,1,o),V=e.Lo(t.f,r)+e.Lo(t.f,i),V>2)){for(h=new X.uB(t.f,V,false),L=0,g=0;g<e.Lo(t.f,r);g++)Q=e.Zo(t.f,r,g),Q!=i&&(h.e[L++]=Q,t.b[Q]=true,t.c[e._o(t.f,r,g)]=true);for(h.e[L++]=r,h.e[L++]=i,B=0;B<e.Lo(t.f,i);B++)Q=e.Zo(t.f,i,B),Q!=r&&(h.e[L++]=Q,t.b[Q]=true,t.c[e._o(t.f,i,B)]=true);for(c=0;c<V;c++)h.a[c]=c,h.b[c]=0,h.k[c]=1;t.b[r]=true,t.b[i]=true,t.c[o]=true,X.FA(t.d,h);}for(D=0;D<t.f.r;D++)if(!t.c[D]&&e.fm(t.f,D)==2){for(A=j(l.j7,H,6,t.f.q,15,1),c=0;c<2;c++)if(A[0]=e.Zl(t.f,c,D),A[1]=e.Zl(t.f,1-c,D),e.Qo(t.f,A[0])==1&&e.Qo(t.f,A[1])==2&&e.Lo(t.f,A[1])==2){t.b[A[0]]=true,t.b[A[1]]=true,t.c[D]=true,S=1;do{if(I=e.Zo(t.f,A[S],0)==A[S-1]?1:0,A[S+1]=e.Zo(t.f,A[S],I),e.Qo(t.f,A[S+1])==2&&e.Lo(t.f,A[S+1])>2)break;t.b[A[S+1]]=true,t.c[e._o(t.f,A[S],I)]=true,++S;}while(e.Qo(t.f,A[S])==2&&e.Lo(t.f,A[S])==2);for(V=e.Lo(t.f,A[0])+e.Lo(t.f,A[S])+S-1,h=new X.uB(t.f,V,false),p=0;p<=S;p++)h.a[p]=p,h.b[p]=0,h.k[p]=64,h.e[p]=A[p];for(s=S+1,b=false,q=0;q<e.Lo(t.f,A[0]);q++)Q=e.Zo(t.f,A[0],q),Q!=A[1]&&(h.a[s]=-0.5,h.b[s]=b?d.Math.sin(b0):-d.Math.sin(b0),h.k[s]=64,h.e[s]=Q,++s,b=true);for(b=false,M=0;M<e.Lo(t.f,A[S]);M++)Q=e.Zo(t.f,A[S],M),Q!=A[S-1]&&(h.a[s]=S+.5,h.b[s]=b?-d.Math.sin(b0):d.Math.sin(b0),h.k[s]=64,h.e[s]=Q,++s,b=true);X.FA(t.d,h);}}for(n=0;n<t.f.q;n++)if(e.Lo(t.f,n)==4){for(Z=j(l.j7,H,6,4,15,1),z=j(l.j7,H,6,4,15,1),tt=0,g=0;g<4;g++)Z[tt]=e.Zo(t.f,n,g),z[tt]=e._o(t.f,n,g),e.Lo(t.f,Z[tt])==1&&!t.c[z[tt]]&&++tt;if(tt==2){for(h=new X.uB(t.f,3,false),c=0;c<2;c++)t.b[Z[c]]=true,t.c[z[c]]=true,h.e[c]=Z[c],h.k[c]=32;h.a[0]=-0.5,h.b[0]=.866,h.a[1]=.5,h.b[1]=.866,h.a[2]=0,h.b[2]=0,h.k[2]=32,h.e[2]=n,X.FA(t.d,h);}if(tt==3){for(B=0;B<2;B++)e.fm(t.f,z[B])==1&&(lt=Z[B],Z[B]=Z[2],Z[2]=lt,lt=z[B],z[B]=z[2],z[2]=lt);for(h=new X.uB(t.f,4,false),c=0;c<3;c++)t.b[Z[c]]=true,t.c[z[c]]=true,h.e[c]=Z[c],h.k[c]=32;h.a[0]=-1,h.b[0]=0,h.a[1]=1,h.b[1]=0,h.a[2]=0,h.b[2]=1,h.a[3]=0,h.b[3]=0,h.k[3]=32,h.e[3]=n,X.FA(t.d,h);}}},X.uA=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;for(u=0,n=0;n<t.f.q;n++)e.Km(t.f,n)&&++u;if(!(u<2)){for(s=0,Q=0,L=0;L<t.f.r;L++)P=e.Zl(t.f,0,L),r=e.Zl(t.f,1,L),e.Km(t.f,P)&&e.Km(t.f,r)&&(t.c[L]=true,t.b[P]=true,t.b[r]=true,Q+=e.dm(t.f,L),++s);for(s!=0&&Q!=0?Q/=s:Q=e.Ul(t.f),i=0;i<t.f.q;i++)e.Km(t.f,i)&&!t.b[i]&&--u;if(!(u<2)){for(p=j(l.j7,H,6,t.f.q,15,1),h=e.hp(t.f,p,true,true),M=j(l.j7,H,6,h,15,1),D=0;D<t.f.q;D++)p[D]!=-1&&++M[p[D]];for(k=j(l.H8,C1,47,h,0,1),c=0;c<h;c++)k[c]=new X.uB(t.f,M[c],true);for(o=j(l.j7,H,6,h,15,1),A=0;A<t.f.q;A++)g=p[A],g!=-1&&(k[g].k[o[g]]=i0,k[g].e[o[g]]=A,k[g].a[o[g]]=e.Ql(t.f,A)/Q,k[g].b[o[g]]=e.Rl(t.f,A)/Q,++o[g]);for(q=-1,S=0,B=0;B<h;B++)S<M[B]&&(S=M[B],q=B);for(X.FA(t.d,k[q]),b=0;b<h;b++)b!=q&&X.FA(t.d,k[b]);}}},X.vA=function(t){var A,n;for(A=0;A<t.f.q;A++)!t.b[A]&&e.Lo(t.f,A)==0&&(n=new X.uB(t.f,1,false),t.b[A]=true,n.e[0]=A,n.a[0]=0,n.b[0]=0,n.k[0]=0,X.FA(t.d,n));},X.wA=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;for(c=null,c=new e.iu(),e.cu(c,t.f),D=j(l.Feb,ht,6,t.f.f,16,1),B=A.Tb();B.dd();)if(g=B.ed(),b=null,M=g.b,e._t(c,M),e.Pt(c,4,8)!=0&&(b=c.F),b)for(h=new y.Bqb(b);h.a<h.c.a.length;){for(s=y.Aqb(h),k=0,P=s,r=0,i=P.length;r<i;++r)n=P[r],D[n]&&++k;if(k<=1){for(o=g.c,t.a?o=false:t.a=true,Q=new X.uB(t.f,s.length,o),L=0;L<s.length;L++)n=s[L],o&&e.sn(t.f,n,true),Q.k[L]=256,Q.e[L]=n,Q.a[L]=e.Ql(g.b,L)/g.a,Q.b[L]=e.Rl(g.b,L)/g.a,D[n]=true,t.b[n]=true;for(u=0;u<M.g;u++)t.c[e.Xo(t.f,s[M.D[0][u]],s[M.D[1][u]])]=true;X.FA(t.d,Q);}}return D;},X.xA=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q;for(A=X.Rz(t),i=j(l.f7,Y,6,t.f.r,15,1),X.sA(t,i,A),r=0;r<t.f.r;r++)i[r]==2&&(e.fq(t.f,e.Zl(t.f,0,r))||e.fq(t.f,e.Zl(t.f,1,r)))&&(i[r]=3);for(b=0;b<t.d.a.length;b++){for(s=y.xj(t.d,b),o=X.lB(s),k=s.c,B=new X.vB(s),g=-1,h=0;h<224&&o.a.length!=0;h++){if(Q=y.Avb(t.g,o.a.length),u=(w.Pzb(Q,o.a.length),o.a[Q]),D=X.gA(t,u[0],u[1]),n=j(l.j7,H,6,D.length,15,1),P=0,h<32)for(c=1;c<D.length-1;c++)i[D[c]]==3&&(n[P++]=D[c]);else if(h<96)for(c=1;c<D.length-1;c++)i[D[c]]>=2&&(n[P++]=D[c]);else for(c=1;c<D.length-1;c++)i[D[c]]>=1&&(n[P++]=D[c]);if(P!=0){if(p=n[0],P>1)do p=n[y.Avb(t.g,P)];while(p==g);p!=g&&(g=p,X.iB(s,p),o=X.lB(s),k>s.c&&(k=s.c,B=new X.vB(s)));}}y.Hj(t.d,b,B),s=B,L=1;do{for(M=9999,c=0;c<s.e.length;c++)q=A[s.e[c]],q==L?X.rB(s,c):q>L&&q<M&&(M=q);L=M;}while(M!=9999);}},X.yA=function(t,A){t.g=new y.Evb(A);},X.zA=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p;for(r=j(l.h7,Ft,6,e.Lo(t.f,n)+1,15,1),D=j(l.j7,H,6,e.Lo(t.f,n)+1,15,1),u=j(l.j7,H,6,e.Lo(t.f,n)+1,15,1),B=X.nB(A,n),i=0,Q=0;Q<e.Lo(t.f,n);Q++)D[i]=e.Zo(t.f,n,Q),u[i]=e._o(t.f,n,Q),s=X.nB(A,D[i]),s!=-1&&(r[i++]=X.$A(A.a[B],A.b[B],A.a[s],A.b[s]));if(i==1)return r[0]+A1;for(L=i-1;L>0;L--)for(h=0;h<L;h++)r[h]>r[h+1]&&(k=r[h],r[h]=r[h+1],r[h+1]=k,M=D[h],D[h]=D[h+1],D[h+1]=M,p=u[h],u[h]=u[h+1],u[h+1]=p);for(r[i]=r[0]+I1,D[i]=D[0],u[i]=u[0],b=-100,c=0,o=0;o<i;o++)P=r[o+1]-r[o],i>2&&e.gq(t.f,u[o])&&e.gq(t.f,u[o+1])&&(g=X.iA(t,D[o],n,D[o+1]),g!=0&&(P-=100-g)),b<P&&(b=P,c=o);return (r[c]+r[c+1])/2;},X.AA=function(t,A,n,P){var r;r=d.Math.min(y.zj(t.d,A,0),y.zj(t.d,n,0)),y.rj(t.d,r,P),y.Fj(t.d,A),y.Fj(t.d,n);},X.BA=function(){X.CA.call(this,2);},X.CA=function(t){this.e=t,(t&1)==0&&!X.Oz&&!X.Oz&&(X.Oz=new X.dB());},X.DA=function(t,A){var n,P,r,i,D;for(D=0,n=0,P=0;P<A;P++)D+=t[P].b*d.Math.sin(t[P].a),n+=t[P].b*d.Math.cos(t[P].a);return n==0?i=D>0?k0:un:(i=d.Math.atan(D/n),n<0&&(i+=A1)),r=d.Math.sqrt(D*D+n*n)/A,new X.YA(i,r);},X.EA=function(t,A){var n,P;return n=d.Math.abs(t.b),P=d.Math.abs(A.b),n<P?-1:n>P?1:0;},N(68,1,{},X.BA,X.CA),U.a=false,U.e=0,l.B8=O(68),X.FA=function(t,A){var n,P;for(P=new y.Bqb(t);P.a<P.c.a.length;)if(n=y.Aqb(P),X.gB(n,A))return  false;return w.szb(t.a,A),true;},X.GA=function(t,A){return X.FA(t,A);},X.HA=function(){y.Mj.call(this);},N(308,20,Wn,X.HA),U.add=function(t){return X.GA(this,t);},l.x8=O(308),X.JA=function(t){var A,n;return A=t.c.e.length,n=t.c.e.length,n<A?-1:n>A?1:0;},X.KA=function(){},N(310,1,{},X.KA),U.Rb=function(t,A){var n;return X.JA((n=t,n));},U.mb=function(t){return this===t;},l.y8=O(310),X.NA=function(t){var A,n;return A=t.c.e.length,n=t.c.e.length,A<n?-1:A>n?1:0;},X.OA=function(){},N(311,1,{},X.OA),U.Rb=function(t,A){var n;return X.NA((n=t,n));},U.mb=function(t){return this===t;},l.z8=O(311),X.RA=function(){},N(309,1,{},X.RA),U.Rb=function(t,A){return X.EA(t,A);},U.mb=function(t){return this===t;},l.A8=O(309),X.UA=function(t,A,n){t.c[0]+=X.jB(t.b[0],A),t.d[0]+=X.kB(t.b[0],A),t.c[1]+=X.jB(t.b[1],n),t.d[1]+=X.kB(t.b[1],n),++t.a[0],++t.a[1];},X.VA=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V;for(P=j(l.h7,Ft,6,2,15,1),Q=0;Q<2;Q++)t.c[Q]/=t.a[Q],t.d[Q]/=t.a[Q],P[Q]=X.eB(t.b[Q],t.c[Q],t.d[Q],t.b[1-Q].e.length,A);for(X.sB(t.b[0],t.c[0],t.d[0],k0-P[0]),X.sB(t.b[1],t.c[1],t.d[1],4.71238898038469-P[1]),V=U0,S=or,D=t.d[0]-t.d[1],L=0;L<t.b[1].b.length;L++)t.b[1].b[L]+=D,V>t.b[1].b[L]&&(V=t.b[1].b[L]),S<t.b[1].b[L]&&(S=t.b[1].b[L]);for(p=S-V+2*A,r=Rt(d.Math.ceil(p)),V+=(p-r)/2-A,k=j(l.h7,Ft,6,r,15,1),s=0;s<r;s++)k[s]=t.c[1]+A;for(h=0;h<t.b[1].b.length;h++)for(q=t.b[1].b[h]-V,M=Rt(q-A),u=d.Math.min(Rt(q+A),r-1),B=M;B<=u;B++)k[B]>t.b[1].a[h]&&(k[B]=t.b[1].a[h]);for(b=0;b<r;b++)k[b]-=A;for(i=t.c[0]-t.c[1],c=0;c<t.b[0].a.length;c++)g=Rt(t.b[0].b[c]-V),g>=0&&g<k.length&&i<t.b[0].a[c]-k[g]&&(i=t.b[0].a[c]-k[g]);for(o=0;o<t.b[1].a.length;o++)t.b[1].a[o]+=i;n&&(X.sB(t.b[0],t.c[0],t.d[0],P[0]-k0),X.sB(t.b[1],t.c[0],t.d[0],P[0]-k0));},X.WA=function(t,A){var n,P;for(this.b=j(l.H8,C1,47,2,0,1),this.b[0]=t,this.b[1]=A,this.c=j(l.h7,Ft,6,2,15,1),this.d=j(l.h7,Ft,6,2,15,1),this.a=j(l.j7,H,6,2,15,1),n=0;n<2;n++){for(P=0;P<this.b[n].e.length;P++)this.c[n]+=X.jB(this.b[n],P),this.d[n]+=X.kB(this.b[n],P);this.a[n]=this.b[n].e.length;}},X.XA=function(t,A,n,P){this.b=j(l.H8,C1,47,2,0,1),this.b[0]=t,this.b[1]=A,this.c=j(l.h7,Ft,6,2,15,1),this.d=j(l.h7,Ft,6,2,15,1),this.c[0]=X.jB(this.b[0],n),this.d[0]=X.kB(this.b[0],n),this.c[1]=X.jB(this.b[1],P),this.d[1]=X.kB(this.b[1],P),this.a=j(l.j7,H,6,2,15,1),this.a[0]=1,this.a[1]=1;},N(106,1,{106:1},X.WA,X.XA),l.C8=O(106),X.YA=function(t,A){this.a=t,this.b=A;},X.ZA=function(t,A,n,P){var r,i;this.a=X.$A(t,A,n,P),r=n-t,i=P-A,this.b=d.Math.sqrt(r*r+i*i);},X.$A=function(t,A,n,P){var r,i,D;return i=n-t,D=P-A,D!=0?(r=d.Math.atan(i/D),D<0&&(i<0?r-=A1:r+=A1)):r=i>0?k0:un,r;},N(41,1,{41:1},X.YA,X.ZA),U.a=0,U.b=0,l.D8=O(41),X._A=function(t){this.a=j(l.j7,H,6,t,15,1),this.b=j(l.j7,H,6,t,15,1);},N(247,1,{},X._A),l.E8=O(247),X.aB=function(t,A,n){this.c=t,this.a=A,this.b=n;},N(197,1,{197:1},X.aB),U.a=0,U.b=0,l.F8=O(197),X.cB=function(){X.cB=D1,X.bB=v(F(l.Ybb,1),st,2,6,["gkvt@@@@LddTTTrbTRTRTRRRRRRRRRRRRRrVRrIh\\IAaQxlY@gRHdJCJcRXlv_CfJx|A\\hRHejiLaQjTje^kSjtFcIhvXmVKMjt{lN{Kavy\\^wGjjjjjjjjjjjjjjjjjjjjjjjjjjjjjjh@@vo@HBC@PhLN@bPhtFKCcpDbILaRhtzCIbsX\\nOO`JDbqDjSKdJeJmQjtz}Ahr[LVkMnpz\\nwGj{PBhBdBlBBBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajijej` !BnkjyVwsVr|iQn|Q|goTZWPIJwbudnRkVYBez]siZymNJZUqNFBqZWxS~iCXVU]SeRjwrtSPAjkvXLpBAZauDPzq]PfMlecrMnkv|@\\SFD`m|mWiEoCXp`SIe_J[l|[XCbloTV`[Gc@FJGopyyoOlFQfUy^w\\Bgz|","gcrt@@@@LdbbbbTRbRbRbRRRRRRRRRRRRVRrVQIA`HtRGAaIxZAHfShTjCIbqylQGKgqdBaXeQJeruBiPitZmFoPZLFSYbvZlVGMnsZ]vWSmr{]UUUUUUUUUUUUUUUUUUUUUUUUUUUUUT@@[G`DAA`HTFG@QHTZCEaqxBQDfPiTZ]AdqYlNWGgpEBQXbUIerEReVhuZ]^`tYMfKUfwX]NW[jkPBhBdBlBbBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajij` !B^cR]`]Fm]QkfljE\\p\x7FUVfgOmFXsQe_gXPyXis_wF|vUUX_XbxpzU]HUFgYViwFo~@uemc@}~T\x7FIEPioYVwr]JnM~[ZEC\\g}~o_pUfdo~irsklTLiyVJshnw^iVAsZ`_~}PYkckURH{FYMImFaRaccUlCZSHMfP","dml@@Dje^VGiyZjjjh@vtHSBinFU@ !BPTCTy[skMzUPF`AJbBixEZHS[Il","dml@@DjYVvGiyZjjjh@vtHSBinFU@ !BwLo~BJ~UquhXBinZ\\ykA@F_eMrT","dml@@LdfbbQX^fUZjjj`C[PaLJfxYT !BzxIHVc{OiJVRpprePho~]}y\x7FwLl","deL@@DjUYkfEijjjj@MeBDpj[ad !B\x7FaA\x7FMVr[AvkKzm_jKvVbD{sk","dil@@LddTQRl[NX^Fjjjj@MiBDpj[a@ !BPfL@\x7Fox@M~T@\x7Fox@\x7F`C~@@","daL@@DjYtKJqjynjjjj@MaBDpj[` !B`bL@_gx@@Gy~@Gx@_`@"]);},X.dB=function(){X.cB();var t,A,n,P,r,i,D;for(y.Mj.call(this),i=new e.yu(),n=X.bB,P=0,r=n.length;P<r;++P)A=n[P],t=e.Pq(new e.Xq(),A),e.ru(i,t),D=new X.wB(t),D.a=e.Ul(D.b),w.szb(this.a,D);},N(362,20,Wn,X.dB),l.G8=O(362),X.eB=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G;if(t.e.length==1)return 0;for(z=r+d.Math.sqrt(P),L=j(l.h7,Ft,6,36,15,1),c=0;c<t.e.length;c++)i=X.$A(A,n,t.a[c],t.b[c]),u=X.fB(Jt(c1(d.Math.round(i*36/I1)))),s=A-t.a[c],h=n-t.b[c],ot=s*s+h*h,L[u]<ot&&(L[u]=ot);for(S=-1,q=-1,g=0;g<36;g++)L[g]=d.Math.sqrt(L[g]),S<L[g]&&(S=L[g],q=g);for(tt=q-18<0?q-18+36:q-18>=36?q-18-36:q-18,B=0;B<=18;B++)L[tt+B<0?tt+B+36:tt+B>=36?tt+B-36:tt+B]+=.01*B,L[tt-B<0?tt-B+36:tt-B>=36?tt-B-36:tt-B]+=.01*B;for(G=j(l.h7,Ft,6,9,15,1),o=j(l.h7,Ft,6,9,15,1),k=1;k<9;k++)G[k]=d.Math.sin(k*P9),o[k]=d.Math.cos(k*P9);for(I=U0,V=-1,D=0;D<36;D++)if(!(L[D]>=I)){for(p=L[D],b=1;b<9;b++){for(M=-1;M<=1&&(Z=D+M*b<0?D+M*b+36:D+M*b>=36?D+M*b-36:D+M*b,!(!(L[Z]*o[b]<=p)&&(Q=o[b]*d.Math.min(L[Z],z/G[b]),p<Q&&(p=Q,I<=p))));M+=2);if(I<=p)break;}I>p&&(I=p,V=D);}return I1*V/36;},X.fB=function(t){return t<0?t+36:t>=36?t-36:t;},X.gB=function(t,A){var n,P,r;if(A.e.length!=t.e.length)return  false;for(P=X.oB(t),r=X.oB(A),n=0;n<P.length;n++)if(P[n]!=r[n])return  false;return  true;},X.hB=function(t,A,n){var P,r,i;for(i=0;i<t.e.length;i++)r=d.Math.sqrt((t.a[i]-A)*(t.a[i]-A)+(t.b[i]-n)*(t.b[i]-n)),P=0-X.$A(A,n,t.a[i],t.b[i]),t.a[i]=A+r*d.Math.sin(P),t.b[i]=n+r*d.Math.cos(P);},X.iB=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V;if(t.d==null&&(t.d=j(l.j7,E1,7,t.j.r,0,2)),t.d[A]==null){for(b=j(l.j7,H,6,t.e.length,15,1),p=j(l.Feb,ht,6,t.j.q,16,1),r=e.Zl(t.j,0,A),i=e.Zl(t.j,1,A),b[0]=r,p[r]=true,L=0,c=0;L<=c;){for(B=0;B<e.Lo(t.j,b[L]);B++)D=e.Zo(t.j,b[L],B),!p[D]&&D!=i&&(b[++c]=D,p[D]=true);if(L==c)break;++L;}if(h=c+1>(t.e.length/2|0),t.i){for(o=false,u=false,B=0;B<t.e.length;B++)P=t.e[B],e.Km(t.j,P)&&P!=r&&P!=i&&(p[t.e[B]]?o=true:u=true);o!=u&&(h=o);}for(Q=2,t.d[A]=j(l.j7,H,6,h?t.e.length-c:c+2,15,1),k=0;k<t.e.length;k++)t.e[k]==r?t.d[A][h?0:1]=k:t.e[k]==i?t.d[A][h?1:0]=k:h^p[t.e[k]]&&(t.d[A][Q++]=k);}for(S=t.a[t.d[A][0]],V=t.b[t.d[A][0]],q=X.$A(S,V,t.a[t.d[A][1]],t.b[t.d[A][1]]),g=2;g<t.d[A].length;g++)M=t.d[A][g],s=d.Math.sqrt((t.a[M]-S)*(t.a[M]-S)+(t.b[M]-V)*(t.b[M]-V)),n=2*q-X.$A(S,V,t.a[M],t.b[M]),t.a[M]=S+s*d.Math.sin(n),t.b[M]=V+s*d.Math.cos(n);},X.jB=function(t,A){return t.a[A];},X.kB=function(t,A){return t.b[A];},X.lB=function(t){var A,n,P,r,i,D,u,o;for(t.c=0,n=new y.Mj(),r=1;r<t.e.length;r++)for(i=0;i<r;i++)u=d.Math.abs(t.a[r]-t.a[i]),o=d.Math.abs(t.b[r]-t.b[i]),P=d.Math.sqrt(u*u+o*o),P<.8&&(A=j(l.j7,H,6,2,15,1),A[0]=t.e[r],A[1]=t.e[i],w.szb(n.a,A)),D=1-d.Math.min(P,1),t.c+=D*D;return n;},X.mB=function(t,A){return t.e[A];},X.nB=function(t,A){var n;for(n=0;n<t.e.length;n++)if(A==t.e[n])return n;return  -1;},X.oB=function(t){return t.n==null&&(t.n=y.Jqb(t.e,t.e.length),w.vzb(t.n,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[])))),t.n;},X.pB=function(t,A){var n;for(n=0;n<t.e.length;n++)if(A==t.e[n])return  true;return  false;},X.qB=function(t){var A,n,P,r,i,D;for(P=0,i=0;i<t.e.length;i++)for(A=t.e[i],n=e.Lo(t.j,A),D=0;D<n;D++)e.Zo(t.j,A,D)>A&&++P;for(t.f=j(l.j7,H,6,P,15,1),t.g=j(l.j7,H,6,t.j.q,15,1),P=0,r=0;r<t.e.length;r++)for(A=t.e[r],n=e.Lo(t.j,A),t.g[A]=r,D=0;D<n;D++)e.Zo(t.j,A,D)>A&&(t.f[P++]=e._o(t.j,A,D));},X.rB=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;for(c=t.a[A],M=t.b[A],i=j(l.D8,C1,41,4,0,1),L=0,s=0;s<t.f.length&&!(L>=4);s++)if(!(A==t.g[e.Zl(t.j,0,t.f[s])]||A==t.g[e.Zl(t.j,1,t.f[s])])){if(g=t.a[t.g[e.Zl(t.j,0,t.f[s])]],p=t.b[t.g[e.Zl(t.j,0,t.f[s])]],B=t.a[t.g[e.Zl(t.j,1,t.f[s])]],q=t.b[t.g[e.Zl(t.j,1,t.f[s])]],u=d.Math.sqrt((g-c)*(g-c)+(p-M)*(p-M)),o=d.Math.sqrt((B-c)*(B-c)+(q-M)*(q-M)),r=d.Math.sqrt((B-g)*(B-g)+(q-p)*(q-p)),u<r&&o<r){g==B?(D=d.Math.abs(c-g),D<.5&&(i[L++]=new X.YA(X.$A(g,M,c,M),(.5-D)/2))):p==q?(D=d.Math.abs(M-p),D<.5&&(i[L++]=new X.YA(X.$A(c,p,c,M),(.5-D)/2))):(h=(q-p)/(B-g),b=-1/h,n=p-h*g,P=M-b*c,k=(P-n)/(h-b),S=h*k+n,D=d.Math.sqrt((k-c)*(k-c)+(S-M)*(S-M)),D<.5&&(i[L++]=new X.YA(X.$A(k,S,c,M),(.5-D)/2)));continue;}if(u<.5){i[L++]=new X.YA(X.$A(g,p,c,M),(.5-u)/2);continue;}if(o<.5){i[L++]=new X.YA(X.$A(B,q,c,M),(.5-o)/2);continue;}}L>0&&(Q=X.DA(i,L),t.a[A]+=Q.b*d.Math.sin(Q.a),t.b[A]+=Q.b*d.Math.cos(Q.a));},X.sB=function(t,A,n,P){var r,i,D;for(D=0;D<t.e.length;D++)i=d.Math.sqrt((t.a[D]-A)*(t.a[D]-A)+(t.b[D]-n)*(t.b[D]-n)),r=X.$A(A,n,t.a[D],t.b[D])+P,t.a[D]=A+i*d.Math.sin(r),t.b[D]=n+i*d.Math.cos(r);},X.tB=function(t,A,n){var P;for(P=0;P<t.e.length;P++)t.a[P]+=A,t.b[P]+=n;},X.uB=function(t,A,n){this.j=t,this.i=n,this.e=j(l.j7,H,6,A,15,1),this.k=j(l.j7,H,6,A,15,1),this.a=j(l.h7,Ft,6,A,15,1),this.b=j(l.h7,Ft,6,A,15,1);},X.vB=function(t){var A,n;for(this.j=t.j,this.i=t.i,this.e=j(l.j7,H,6,t.e.length,15,1),this.k=j(l.j7,H,6,t.e.length,15,1),this.a=j(l.h7,Ft,6,t.e.length,15,1),this.b=j(l.h7,Ft,6,t.e.length,15,1),n=0;n<t.e.length;n++)this.e[n]=t.e[n],this.k[n]=t.k[n],this.a[n]=t.a[n],this.b[n]=t.b[n];if(t.f!=null)for(this.f=j(l.j7,H,6,t.f.length,15,1),A=0;A<t.f.length;A++)this.f[A]=t.f[A];if(t.g!=null)for(this.g=j(l.j7,H,6,t.g.length,15,1),A=0;A<t.g.length;A++)this.g[A]=t.g[A];},N(47,1,{47:1},X.uB,X.vB),U.c=0,U.i=false,l.H8=O(47),X.wB=function(t){this.b=t,this.c=false;},N(246,1,{246:1},X.wB),U.a=0,U.c=false,l.I8=O(246),gA.xB=function(t){t.c=new y.Mj();},gA.yB=function(t,A,n,P,r,i){var D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot;for(p=-1,I=0,V=0,z=0,B=0,k=0,M=0,tt=0,ot=0,I=0,L=0;L<t.d;L++)I+=P[L]*P[L];if(I=d.Math.sqrt(I),I>i)for(s=0;s<t.d;s++)P[s]*=i/I;for(V=0,h=0;h<t.d;h++)V+=P[h]*t.e[h];if(V>=0)return p;for(z=0,b=0;b<t.d;b++)Z=d.Math.abs(P[b])/d.Math.max(d.Math.abs(A[b]),1),Z>z&&(z=Z);for(M=1e-7/z,B=1,g=0;g<1e3;){if(B<M){p=1;break;}for(c=0;c<t.d;c++)r[c]=A[c]+B*P[c];if(t.k=K.xC(t,r),t.k-n<=wn*B*V)return 0;g==0?tt=-V/(2*(t.k-n-V)):(q=t.k-n-B*V,S=ot-n-k*V,D=(q/(B*B)-S/(k*k))/(B-k),u=(-k*q/(B*B)+B*S/(k*k))/(B-k),D==0?tt=-V/(2*u):(o=u*u-3*D*V,o<0?tt=.5*B:u<=0?tt=(-u+d.Math.sqrt(o))/(3*D):tt=-V/(u+d.Math.sqrt(o))),tt>.5*B&&(tt=.5*B)),k=B,ot=t.k,B=d.Math.max(tt,.1*B),++g;}for(Q=0;Q<t.d;Q++)r[Q]=A[Q];return p;},gA.zB=function(t,A,n){var P,r,i,D;for(D=1,r=0;r<t.g.q;r++)t.i[3*r]=e.Ql(t.g,r),t.i[3*r+1]=e.Rl(t.g,r),t.i[3*r+2]=e.Sl(t.g,r);if(D=gA.AB(t,A,n),D==0)for(P=0;P<t.g.q;P++)e.zn(t.g,P,t.i[3*P]),e.An(t.g,P,t.i[3*P+1]),e.Bn(t.g,P,t.i[3*P+2]);for(i=new y.Bqb(t.c);i.a<i.c.a.length;)y.Aqb(i),null.yd();return D;},gA.AB=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt;for(t.e=j(l.h7,Ft,6,t.d,15,1),P=j(l.h7,Ft,6,t.d,15,1),Q=j(l.h7,Ft,6,t.d,15,1),Z=j(l.h7,Ft,6,t.d,15,1),wt=j(l.h7,Ft,6,t.d,15,1),M=j(l.h7,Ft,6,t.d*t.d,15,1),s=0;s<t.d;s++)Z[s]=t.i[s];for(u=K.xC(t,t.i),K.yC(t),tt=0,h=0;h<t.d;h++)M[h*t.d+h]=1,wt[h]=-t.e[h],tt+=t.i[h]*t.i[h];for(I=100*d.Math.max(d.Math.sqrt(tt),t.d),it=(T.Zjb(),c1(Date.now())),q=1;q<=A&&!t.f;q++){if(z=gA.yB(t,t.i,u,wt,Z,I),z<0)return 2;for(u=t.k,lt=0,b=0;b<t.d;b++)wt[b]=Z[b]-t.i[b],t.i[b]=Z[b],ut=d.Math.abs(wt[b])/d.Math.max(d.Math.abs(t.i[b]),1),ut>lt&&(lt=ut),P[b]=t.e[b];if(lt<12e-8)return 0;for(o=K.yC(t),lt=0,Dt=d.Math.max(t.k*o,1),c=0;c<t.d;c++)yt=d.Math.abs(t.e[c])*d.Math.max(d.Math.abs(t.i[c]),1),lt=d.Math.max(lt,yt),P[c]=t.e[c]-P[c];if(lt/=Dt,lt<n)return 0;for(r=0,D=0,ot=0,G=0,g=0;g<t.d;g++){for(p=g*t.d,Q[g]=0,S=0;S<t.d;S++)Q[g]+=M[p+S]*P[S];r+=P[g]*wt[g],D+=P[g]*Q[g],ot+=P[g]*P[g],G+=wt[g]*wt[g];}if(r>d.Math.sqrt(3e-8*ot*G)){for(r=1/r,i=1/D,B=0;B<t.d;B++)P[B]=r*wt[B]-i*Q[B];for(k=0;k<t.d;k++)for(p=k*t.d,S=k;S<t.d;S++)M[p+S]+=r*wt[k]*wt[S]-i*Q[k]*Q[S]+D*P[k]*P[S],M[S*t.d+k]=M[p+S];}for(L=0;L<t.d;L++)for(p=L*t.d,wt[L]=0,S=0;S<t.d;S++)wt[L]-=M[p+S]*t.e[S];if(Qt=c1(Date.now()),ct=N1(Qt,it),ge(ct,t.j)){for(V=new y.Bqb(t.c);V.a<V.c.a.length;)y.Aqb(V),null.yd();it=Qt;}}return 1;},gA.BB=function(t){var A,n,P;for(gA.xB(this),P=0,A=0;A<t.f;A++)P+=e.op(t,A);if(P>0)throw Ut(new T.jib("molecule needs explicit hydrogen atoms for force field calculations"));for(this.g=t,this.d=3*t.q,this.e=j(l.h7,Ft,6,this.d,15,1),this.i=j(l.h7,Ft,6,this.d,15,1),this.f=false,this.j=20,n=0;n<t.q;n++)this.i[3*n]=t.J[n].a,this.i[3*n+1]=t.J[n].b,this.i[3*n+2]=t.J[n].c;},N(275,1,{}),U.d=0,U.f=false,U.j=0,U.k=0,l.J8=O(275),K.CB=function(t,A,n,P,r){this.f=A,this.a=n,this.b=P,this.c=r,this.d=Pt.bE(t.b,A.a[P]),this.g=Pt.WD(t.a,A,n,P,r),this.e=Pt.VD(t.a,A,n,P,r);},K.DB=function(t,A){var n,P,r,i,D,u;for(n=new y.Mj(),P=0;P<A.q;P++)if(A.e[P]>1)for(r=0;r<A.e[P];r++)for(D=A.i[P][r],i=r+1;i<A.e[P];i++)u=A.i[P][i],y.sj(n,new K.CB(t,A,D,P,u));return n;},N(353,1,Ue,K.CB),U.cc=function(t,A,n,P){var r,i,D;return P&&(e.Tl(this.f,this.a)==1||e.Tl(this.f,this.c)==1)?0:(D=K.DD(new K.PD(t,this.b,this.a),new K.PD(t,this.b,this.c)),r=D*ye-this.g,i=this.d?143.9325*this.e*(1+d.Math.cos(D)):r9*this.e*r*r*(1+-6981317e-9*r),i);},U.dc=function(t,A){var n,P,r,i,D,u,o,Q,L,s;o=K.ID(new K.PD(t,this.b,this.a)),Q=K.ID(new K.PD(t,this.b,this.c)),D=K.HD(new K.PD(t,this.b,this.a)),u=K.HD(new K.PD(t,this.b,this.c)),P=(o.a*Q.a+o.b*Q.b+o.c*Q.c)/(d.Math.sqrt(o.a*o.a+o.b*o.b+o.c*o.c)*d.Math.sqrt(Q.a*Q.a+Q.b*Q.b+Q.c*Q.c)),s=1-P*P,L=gn,s>0&&(L=d.Math.sqrt(s)),n=ye*d.Math.acos(P)-this.g,i=DA*this.e*n*(1+-0.0104719755*n),this.d&&(i=-143.9325*this.e*L),r=v(F(l.h7,1),Ft,6,15,[1/D*(Q.a-P*o.a),1/D*(Q.b-P*o.b),1/D*(Q.c-P*o.c),1/u*(o.a-P*Q.a),1/u*(o.b-P*Q.b),1/u*(o.c-P*Q.c)]),A[3*this.a]+=i*r[0]/-L,A[3*this.a+1]+=i*r[1]/-L,A[3*this.a+2]+=i*r[2]/-L,A[3*this.b]+=i*(-r[0]-r[3])/-L,A[3*this.b+1]+=i*(-r[1]-r[4])/-L,A[3*this.b+2]+=i*(-r[2]-r[5])/-L,A[3*this.c]+=i*r[3]/-L,A[3*this.c+1]+=i*r[4]/-L,A[3*this.c+2]+=i*r[5]/-L;},U.a=0,U.b=0,U.c=0,U.d=false,U.e=0,U.g=0,l.K8=O(353),K.gC=function(t){T.fC.call(this,t);},N(185,33,{185:1,4:1,19:1,33:1,27:1},K.gC),l.L8=O(185),K.hC=function(){T.dC.call(this);},N(315,33,m0,K.hC),l.M8=O(315),K.iC=function(t,A,n){K.jC.call(this,t,A,A.D[0][n],A.D[1][n]);},K.jC=function(t,A,n,P){this.d=A,this.a=n,this.b=P,this.e=Pt.pE(t.d,A,n,P),this.c=Pt.oE(t.d,A,n,P);},K.kC=function(t,A){var n,P;for(P=new y.Mj(),n=0;n<A.r;n++)y.sj(P,new K.iC(t,A,n));return P;},N(237,1,Ue,K.iC),U.cc=function(t,A,n,P){var r,i,D;return P&&(e.Tl(this.d,this.a)==1||e.Tl(this.d,this.b)==1)?0:(i=K.FD(new K.OD(t,this.a),new K.OD(t,this.b)),r=(i-this.e)*(i-this.e),D=71.96625*this.c*r*(1+-2*(i-this.e)+2.3333333333333335*r),D);},U.dc=function(t,A){var n,P,r,i;if(P=K.FD(new K.OD(t,this.a),new K.OD(t,this.b)),r=P-this.e,n=143.9325*this.c*r*(1+-3*r+4.666666666666667*r*r),P>0)for(i=0;i<3;i++)A[3*this.a+i]+=n*(t[3*this.a+i]-t[3*this.b+i])/P,A[3*this.b+i]-=n*(t[3*this.a+i]-t[3*this.b+i])/P;},U.a=0,U.b=0,U.c=0,U.e=0,l.N8=O(237),K.nC=function(t){var A,n,P,r,i,D,u,o,Q;A=null;try{for(A=new Ct.ogb(new Ct.tgb((r=_t.mAb(t),S1.Mmb(),r))),o=T.Whb(T.Cjb(Ct.ngb(A)),P1,St),P=T.yjb(T.Cjb(Ct.ngb(A)),","),Q=y1(l.Sbb,[c0,C1],[8,1],5,[o,P.length],2),D=0;(i=Ct.ngb(A))!=null&&D<o;D++)for(u=T.yjb(T.Cjb(i),","),n=0;n<P.length;n++)switch(T.cjb(P[n],0)){case 105:$1(Q[D],n,T.Aib(T.Whb(T.Cjb(u[n]),P1,St)));break;case 102:$1(Q[D],n,w.aAb(T.Vhb(T.Cjb(u[n]))));break;case 99:$1(Q[D],n,P2(T.cjb(T.wjb(T.wjb(T.Cjb(u[n]),"'"),'"'),0)));}return Q;}catch(L){if(L=M1(L),Kt(L,75))T.Zjb();else throw Ut(L);}return y1(l.Sbb,[c0,C1],[8,1],5,[1,1],2);},K.oC=function(t){var A,n,P,r,i,D;for(D=K.nC(t),i=D.length,A=D[0].length,r=y1(l.j7,[E1,H],[7,6],15,[i,A],2),n=0;n<i;n++)for(P=0;P<A;P++)r[n][P]=D[n][P].a;return r;},K.pC=function(t,A,n,P,r,i,D,u){this.e=t,this.a=A,this.b=n,this.f=P,this.d=D,this.c=r*i/u;},K.qC=function(t,A,n,P,r,i){var D,u,o,Q,L;for(u=new y.Mj(),D=r1.CF(t,A),o=0;o<A.q;o++)for(Q=0;Q<o+1;Q++)L=K.ZC(n,new K.iD(o,Q)),(L==(K.eD(),K._C)||L==K.dD)&&d.Math.abs(D[o])>jn&&d.Math.abs(D[Q])>jn&&K.HD(new K.MD(A,o,Q))<P&&y.sj(u,new K.pC(A,o,Q,L,D[o],D[Q],r,i));return u;},N(354,1,Ue,K.pC),U.cc=function(t,A,n,P){var r,i,D;return P&&(e.Tl(this.e,this.a)==1||e.Tl(this.e,this.b)==1)?0:(i=K.HD(new K.PD(t,this.a,this.b)),r=i+.05,this.d&&(r*=r),D=332.0716*this.c/r*(this.f==(K.eD(),K._C)?.75:1),D);},U.dc=function(t,A){var n,P,r,i,D;for(i=K.HD(new K.PD(t,this.a,this.b)),n=i+.05,n*=this.d?n*n:n,P=-332.0716*(this.d?2:1)*this.c/n*(this.f==(K.eD(),K._C)?.75:1),D=0;D<3;D++)r=.02,i>0&&(r=P*(t[3*this.a+D]-t[3*this.b+D])/i),A[3*this.a+D]+=r,A[3*this.b+D]-=r;},U.a=0,U.b=0,U.c=0,U.d=false,l.O8=O(354),K.uC=function(){K.uC=D1,K.tC=new y.ztb();},K.vC=function(t){t.a=new y.Mj();},K.wC=function(t){return K.xC(t,t.i);},K.xC=function(t,A){var n,P,r,i,D;for(D=0,P=1,i=new y.Bqb(t.a);i.a<i.c.a.length;)r=y.Aqb(i),n=r.cc(A,null,""+P,false),n!=0&&(D+=n,++P);return D;},K.yC=function(t){var A,n,P,r,i,D;for(t.e=j(l.h7,Ft,6,t.d,15,1),n=new y.Bqb(t.a);n.a<n.c.a.length;)A=y.Aqb(n),A.dc(t.i,t.e);for(D=-1e8,P=.1,i=0;i<t.d;i++)t.e[i]*=P,t.e[i]>D&&(D=t.e[i]);if(D>10){for(;D*P>10;)P*=.5;for(r=0;r<t.d;r++)t.e[r]*=P;}return P;},K.zC=function(t){K.uC(),K.AC.call(this,t,sP,new y.ztb());},K.AC=function(t,A,n){K.uC();var P,r,i,D,u;gA.BB.call(this,t),K.vC(this),this.b=new K.CC(t),e.bw(this.g,7),u=y.ynb(K.tC,A),i=y.fub(n.c,i9)?T._hb(y.gub(n.c,i9)):100,P=y.fub(n.c,f9)?T._hb(y.gub(n.c,f9)):1,r=y.fub(n.c,D9)&&T.Oc(y.gub(n.c,D9),"distance"),D=new K.$C(this.b),(!y.fub(n.c,u9)||T.Rgb(y.gub(n.c,u9)))&&y.uj(this.a,K.DB(u,this.b)),(!y.fub(n.c,o9)||T.Rgb(y.gub(n.c,o9)))&&y.uj(this.a,K.kC(u,this.b)),(!y.fub(n.c,l9)||T.Rgb(y.gub(n.c,l9)))&&y.uj(this.a,K.qC(u,this.b,D,i,r,P)),(!y.fub(n.c,Q9)||T.Rgb(y.gub(n.c,Q9)))&&y.uj(this.a,K.EC(u,this.b)),(!y.fub(n.c,L9)||T.Rgb(y.gub(n.c,L9)))&&y.uj(this.a,K.oD(u,this.b)),(!y.fub(n.c,U9)||T.Rgb(y.gub(n.c,U9)))&&y.uj(this.a,K.uD(u,this.b)),(!y.fub(n.c,y9)||T.Rgb(y.gub(n.c,y9)))&&y.uj(this.a,K.AD(u,this.b,D,i));},K.BC=function(t,A){K.uC(),y.znb(K.tC,t)||y.Bnb(K.tC,t,A);},N(178,275,{},K.zC,K.AC),l.P8=O(178),K.CC=function(t){e.Rk();var A,n,P,r,i;for(e.vw.call(this,t),e.kp(this,e.Ko(this)),i=(e.bw(this,7),this.p),this.b=j(l.S8,H,6,i.j.a.length,15,1),y.Pqb(this.b),A=false,n=true;!A&&n;)for(A=true,n=false,r=0;r<i.j.a.length;r++)this.b[r]==2&&(this.b[r]=r1.zF(this,r),this.b[r]!=2&&(n=true)),this.b[r]==2&&(A=false);if(!A)throw Ut(new K.hC());for(this.a=j(l.j7,H,6,this.q,15,1),P=0;P<this.a.length;P++)if(this.a[P]=-1,this.a[P]=r1.tF(this,P),this.a[P]==0)throw Ut(new K.gC("Couldn't assign an atom type to atom "+P+" ("+e.Lk[this.C[P]]+")"));},N(276,26,fi,K.CC),l.Q8=O(276),K.DC=function(t,A,n,P,r,i){this.f=A,this.d=n,this.a=P,this.b=r,this.c=i,this.e=Pt.QE(t.k,A,n,P,r,i);},K.EC=function(t,A){var n,P,r,i,D;for(D=new y.Mj(),i=0;i<A.q;i++)A.e[i]==3&&(n=A.i[i][0],P=A.i[i][1],r=A.i[i][2],y.sj(D,new K.DC(t,A,i,n,P,r)),y.sj(D,new K.DC(t,A,i,n,r,P)),y.sj(D,new K.DC(t,A,i,P,r,n)));return D;},N(192,1,Ue,K.DC),U.cc=function(t,A,n,P){var r,i,D,u,o,Q;return P&&(e.Tl(this.f,this.a)==1||e.Tl(this.f,this.b)==1||e.Tl(this.f,this.c)==1)?0:(u=K.ID(new K.PD(t,this.d,this.a)),o=K.ID(new K.PD(t,this.d,this.b)),Q=K.ID(new K.PD(t,this.d,this.c)),D=K.ID(new K.KD(u.b*o.c-u.c*o.b,u.c*o.a-u.a*o.c,u.a*o.b-u.b*o.a)),r=ye*d.Math.asin(D.a*Q.a+D.b*Q.b+D.c*Q.c),i=r9*this.e*r*r,i);},U.dc=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z;for(h=new K.PD(t,this.d,this.a),b=new K.PD(t,this.d,this.b),c=new K.PD(t,this.d,this.c),u=d.Math.sqrt(h.a*h.a+h.b*h.b+h.c*h.c),o=d.Math.sqrt(b.a*b.a+b.b*b.b+b.c*b.c),Q=d.Math.sqrt(c.a*c.a+c.b*c.b+c.c*c.c),h=K.ID(h),b=K.ID(b),c=K.ID(c),s=K.ID(K.ED(new K.KD(-h.a,-h.b,-h.c),b)),g=c.a*s.a+c.b*s.b+c.c*s.c,r=1-g*g,P=d.Math.max(r>0?d.Math.sqrt(r):0,gn),n=ye*d.Math.asin(g),i=h.a*b.a+h.b*b.b+h.c*b.c,k=d.Math.max(1-i*i,gn),B=d.Math.max(k>0?d.Math.sqrt(k):0,gn),D=DA*this.e*n,M=new K.KD(c.b*b.c-c.c*b.b,c.c*b.a-c.a*b.c,c.a*b.b-c.b*b.a),p=new K.KD(h.b*c.c-h.c*c.b,h.c*c.a-h.a*c.c,h.a*c.b-h.b*c.a),q=new K.KD(b.b*h.c-b.c*h.b,b.c*h.a-b.a*h.c,b.a*h.b-b.b*h.a),S=P*B,V=g/(P*k),I=v(F(l.h7,1),Ft,6,15,[(M.a/S-(h.a-b.a*i)*V)/u,(M.b/S-(h.b-b.b*i)*V)/u,(M.c/S-(h.c-b.c*i)*V)/u]),Z=v(F(l.h7,1),Ft,6,15,[(p.a/S-(b.a-h.a*i)*V)/o,(p.b/S-(b.b-h.b*i)*V)/o,(p.c/S-(b.c-h.c*i)*V)/o]),z=v(F(l.h7,1),Ft,6,15,[(q.a/S-c.a*g/P)/Q,(q.b/S-c.b*g/P)/Q,(q.c/S-c.c*g/P)/Q]),L=0;L<3;L++)A[3*this.a+L]+=D*I[L],A[3*this.d+L]+=-D*(I[L]+Z[L]+z[L]),A[3*this.b+L]+=D*Z[L],A[3*this.c+L]+=D*z[L];},U.a=0,U.b=0,U.c=0,U.d=0,U.e=0,l.R8=O(192),K.HC=function(t){return t<3?0:t<11?1:t<19?2:t<37?3:t<55?4:0;},K.IC=function(t){var A;return A=0,t==2?A=1:t>=3&&t<=10?A=2:t>=11&&t<=18?A=3:t>=19&&t<=36?A=4:t>=37&&t<=54&&(A=5),(t>=21&&t<=30||t>=39&&t<=48)&&(A*=10),A;},T.JC=function(t,A){return t.b-A.b;},T.KC=function(t){return t.a!=null?t.a:""+t.b;},T.LC=function(t){return t.b;};function vn(f,t){this.a=f,this.b=t;}N(56,1,{4:1,32:1,56:1}),U.Sb=function(t){return T.JC(this,t);},U.compareTo=function(t){return T.JC(this,t);},U.equals=function(t){return this===t;},U.mb=function(f){return this.equals(f);},U.hashCode=function(){return w.Izb(this);},U.ob=function(){return this.hashCode();},U.name=function(){return T.KC(this);},U.ordinal=function(){return T.LC(this);},U.toString=function(){return this.a!=null?this.a:""+this.b;},U.pb=function(){return this.toString();},U.b=0,l.Gbb=O(56),K.UC=function(){return v(F(l.S8,1),H,6,15,[0,1,2]);},l.S8=GA(null,K.UC),K.VC=function(t,A,n,P,r,i){var D,u,o,Q;if(n=d.Math.max(0,n),P=d.Math.min(i.lc(),P),D=(P-n)/2|0,o=n,u=P,i.kc(o,t)>A||i.kc(u-1,t)<A)return  -1;for(;P>=n;){if(D=n+((P-n)/2|0),Q=i.kc(D,t),Q==A){if(r&&D>o&&i.kc(D-1,t)==A)P=D;else if(!r&&D<u-1&&i.kc(D+1,t)==A)n=D;else return D;}else Q>A?P=D:Q<A&&(n=D);if(P-n==1&&i.kc(n,t)<A&&i.kc(P,t)>A)break;}return  -1;},K.WC=function(t,A){return K.VC(0,t,0,A.a.length,true,A);},K.XC=function(t,A,n){var P,r,i;if(t.length!=A.length)return  -1;for(i=0,P=n.lc(),r=0;r<t.length-1;r++)if(i=K.VC(t[r],A[r],i,P+1,true,n),P=K.VC(t[r],A[r],i,P+1,false,n),i==-1||P==-1)return  -1;return K.VC(t[r],A[r],i,P+1,true,n);},K.YC=function(t){t.a=new y.ztb();},K.ZC=function(t,A){return y.xnb(t.a,A)!=null?y.xnb(t.a,A):(K.eD(),K.dD);},K.$C=function(t){var A,n,P,r,i,D,u,o,Q,L;for(K.YC(this),A=0;A<t.q;A++)for(y.Anb(this.a,new K.iD(A,A),(K.eD(),K.aD)),n=0;n<t.e[A];n++)for(o=t.i[A][n],u=new K.iD(A,o),y.Anb(this.a,u,K.cD),P=0;P<t.e[o];P++)for(Q=t.i[o][P],D=new K.iD(A,Q),(!y.tnb(this.a,D)||F1(y.xnb(this.a,D))===F1(K._C))&&y.Anb(this.a,D,K.bD),r=0;r<t.e[Q];r++)L=t.i[Q][r],i=new K.iD(A,L),y.tnb(this.a,i)||y.Anb(this.a,i,K._C);},N(277,1,{},K.$C),l.U8=O(277),K.eD=function(){K.eD=D1,K.aD=new K.fD("ONE_ONE",0),K.cD=new K.fD("ONE_TWO",1),K.bD=new K.fD("ONE_THREE",2),K._C=new K.fD("ONE_FOUR",3),K.dD=new K.fD("ONE_X",4);},K.fD=function(t,A){vn.call(this,t,A);},K.gD=function(){return K.eD(),v(F(l.T8,1),s0,108,0,[K.aD,K.cD,K.bD,K._C,K.dD]);},N(108,56,{108:1,4:1,32:1,56:1},K.fD),l.T8=GA(108,K.gD),K.hD=function(t,A){return t.a>A.a?1:t.a<A.a?-1:t.b>A.b?1:t.b<A.b?-1:0;},K.iD=function(t,A){this.a=t>A?t:A,this.b=t>A?A:t;},N(99,1,{99:1,32:1},K.iD),U.Sb=function(t){return K.hD(this,t);},U.mb=function(t){var A;return t===this?true:Kt(t,99)?(A=t,this.a==A.a&&this.b==A.b):false;},U.ob=function(){return this.a^this.b;},U.pb=function(){return this.a+","+this.b;},U.a=0,U.b=0,l.V8=O(99),K.nD=function(t,A,n,P,r){this.f=A,this.a=n,this.b=P,this.c=r,this.j=Pt.WD(t.a,A,n,P,r),this.g=Pt.pE(t.d,A,n,P),this.i=Pt.pE(t.d,A,r,P),this.d=Pt.VE(t.n,A,n,P,r),this.e=Pt.VE(t.n,A,r,P,n);},K.oD=function(t,A){var n,P,r,i,D,u,o,Q;for(Q=new y.Mj(),n=0;n<A.q;n++)if(P=A.a[n],!(A.e[n]<=1&&Pt.bE(t.b,P)))for(i=0;i<A.e[n];i++)for(u=A.i[n][i],D=0;D<A.e[n];D++)o=A.i[n][D],!(u>o)&&K.rD(t,A,u,n,o)&&(r=new K.nD(t,A,u,n,o),(d.Math.abs(r.d)>wn||d.Math.abs(r.e)>wn)&&w.szb(Q.a,r));return Q;},K.rD=function(t,A,n,P,r){var i;return i=A.a[P],!(Pt.bE(t.b,i)||n==r||e.Xo(A,n,P)==-1||e.Xo(A,P,r)==-1);},N(355,1,Ue,K.nD),U.cc=function(t,A,n,P){var r,i,D,u,o;return P&&(e.Tl(this.f,this.a)==1||e.Tl(this.f,this.c)==1)?0:(r=K.HD(new K.PD(t,this.b,this.a)),i=K.HD(new K.PD(t,this.b,this.c)),o=K.DD(new K.PD(t,this.b,this.a),new K.PD(t,this.b,this.c)),u=DA*(o*ye-this.j),D=u*(r-this.g)*this.d+u*(i-this.i)*this.e,D);},U.dc=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B;L=K.HD(new K.PD(t,this.b,this.a)),s=K.HD(new K.PD(t,this.b,this.c)),b=K.ID(new K.PD(t,this.b,this.a)),c=K.ID(new K.PD(t,this.b,this.c)),P=b.a*c.a+b.b*c.b+b.c*c.c,B=1-P*P,g=d.Math.max(B>0?d.Math.sqrt(B):0,gn),n=ye*d.Math.acos(P)-this.j,h=ye*(this.d*(L-this.g)+this.e*(s-this.i)),r=1/L*(c.a-P*b.a),i=1/L*(c.b-P*b.b),D=1/L*(c.c-P*b.c),u=1/s*(b.a-P*c.a),o=1/s*(b.b-P*c.b),Q=1/s*(b.c-P*c.c),A[3*this.a]+=DA*(b.a*this.d*n+r/-g*h),A[3*this.a+1]+=DA*(b.b*this.d*n+i/-g*h),A[3*this.a+2]+=DA*(b.c*this.d*n+D/-g*h),A[3*this.b]+=DA*((-b.a*this.d-c.a*this.e)*n+(-r-u)/-g*h),A[3*this.b+1]+=DA*((-b.b*this.d-c.b*this.e)*n+(-i-o)/-g*h),A[3*this.b+2]+=DA*((-b.c*this.d-c.c*this.e)*n+(-D-Q)/-g*h),A[3*this.c]+=DA*(c.a*this.e*n+u/-g*h),A[3*this.c+1]+=DA*(c.b*this.e*n+o/-g*h),A[3*this.c+2]+=DA*(c.c*this.e*n+Q/-g*h);},U.a=0,U.b=0,U.c=0,U.d=0,U.e=0,U.g=0,U.i=0,U.j=0,l.W8=O(355),K.sD=function(t,A){this.a=new Pt.XD(this),this.e=new Pt.wE(),this.b=new Pt.gE(),this.c=new Pt.lE(),this.d=new Pt.qE(this),this.f=new Pt.DE(),this.i=new Pt.JE(),this.g=new Pt.GE(),this.j=new Pt.NE(),this.k=new Pt.RE(this,t),this.n=new Pt.WE(this),this.o=new Pt.$E(this,A),this.p=new Pt.lF();},N(155,1,{155:1},K.sD),l.X8=O(155),K.tD=function(t,A,n,P,r,i){var D;this.a=n,this.b=P,this.c=r,this.d=i,D=Pt.ZE(t.o,A,n,P,r,i),this.e=D.a,this.f=D.b,this.g=D.c;},K.uD=function(t,A){var n,P,r,i,D,u,o,Q,L;for(L=new y.Mj(),n=0;n<A.q;n++)for(D=0;D<A.e[n];D++)for(P=A.i[n][D],u=0;u<A.e[P];u++)if(r=A.i[P][u],n!=r)for(o=0;o<A.e[r];o++)i=A.i[r][o],!(P==i||n==i)&&i>n&&(Q=new K.tD(t,A,n,P,r,i),(d.Math.abs(Q.e)>.001||d.Math.abs(Q.f)>.001||d.Math.abs(Q.g)>.001)&&w.szb(L.a,Q));return L;},N(356,1,Ue,K.tD),U.cc=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b;return o=new K.PD(t,this.a,this.b),Q=new K.PD(t,this.c,this.b),L=new K.PD(t,this.b,this.c),s=new K.PD(t,this.d,this.c),h=new K.KD(o.b*Q.c-o.c*Q.b,o.c*Q.a-o.a*Q.c,o.a*Q.b-o.b*Q.a),b=new K.KD(L.b*s.c-L.c*s.b,L.c*s.a-L.a*s.c,L.a*s.b-L.b*s.a),D=(h.a*b.a+h.b*b.b+h.c*b.c)/(d.Math.sqrt(h.a*h.a+h.b*h.b+h.c*h.c)*d.Math.sqrt(b.a*b.a+b.b*b.b+b.c*b.c)),r=2*D*D-1,i=D*(2*r-1),u=.5*(this.e*(1+D)+this.f*(1-r)+this.g*(1+i)),u;},U.dc=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;D=v(F(l.$8,1),C1,22,0,[new K.PD(t,this.b,this.a),new K.PD(t,this.b,this.c),new K.PD(t,this.c,this.b),new K.PD(t,this.c,this.d)]),h=v(F(l.$8,1),C1,22,0,[K.ED(D[0],D[1]),K.ED(D[2],D[3])]),P=v(F(l.h7,1),Ft,6,15,[K.HD(h[0]),K.HD(h[1])]),!(d.Math.abs(P[0])<jn||d.Math.abs(P[1])<jn)&&(h[0]=K.ID(h[0]),h[1]=K.ID(h[1]),n=K.GD(h[0],h[1]),L=1-n*n,Q=L>0?d.Math.sqrt(L):0,u=2*Q*n,o=3*Q-4*Q*L,i=.5*(-this.e*Q+2*this.f*u-3*this.g*o),s=-i*(d.Math.abs(Q)<jn?1/n:1/Q),r=v(F(l.h7,1),Ft,6,15,[1/P[0]*(h[1].a-n*h[0].a),1/P[0]*(h[1].b-n*h[0].b),1/P[0]*(h[1].c-n*h[0].c),1/P[1]*(h[0].a-n*h[1].a),1/P[1]*(h[0].b-n*h[1].b),1/P[1]*(h[0].c-n*h[1].c)]),A[3*this.a]+=s*(r[2]*D[1].b-r[1]*D[1].c),A[3*this.a+1]+=s*(r[0]*D[1].c-r[2]*D[1].a),A[3*this.a+2]+=s*(r[1]*D[1].a-r[0]*D[1].b),A[3*this.b]+=s*(r[1]*(D[1].c-D[0].c)+r[2]*(D[0].b-D[1].b)+r[4]*-D[3].c+r[5]*D[3].b),A[3*this.b+1]+=s*(r[0]*(D[0].c-D[1].c)+r[2]*(D[1].a-D[0].a)+r[3]*D[3].c+r[5]*-D[3].a),A[3*this.b+2]+=s*(r[0]*(D[1].b-D[0].b)+r[1]*(D[0].a-D[1].a)+r[3]*-D[3].b+r[4]*D[3].a),A[3*this.c]+=s*(r[1]*D[0].c+r[2]*-D[0].b+r[4]*(D[3].c-D[2].c)+r[5]*(D[2].b-D[3].b)),A[3*this.c+1]+=s*(r[0]*-D[0].c+r[2]*D[0].a+r[3]*(D[2].c-D[3].c)+r[5]*(D[3].a-D[2].a)),A[3*this.c+2]+=s*(r[0]*D[0].b+r[1]*-D[0].a+r[3]*(D[3].b-D[2].b)+r[4]*(D[2].a-D[3].a)),A[3*this.d]+=s*(r[4]*D[2].c-r[5]*D[2].b),A[3*this.d+1]+=s*(r[5]*D[2].a-r[3]*D[2].c),A[3*this.d+2]+=s*(r[3]*D[2].b-r[4]*D[2].a));},U.a=0,U.b=0,U.c=0,U.d=0,U.e=0,U.f=0,U.g=0,l.Y8=O(356),K.xD=function(t,A){var n,P,r,i,D;return i=Pt.kF(A.p,t.b),D=Pt.kF(A.p,t.d),n=Pt.gF(A.p,t.b),P=Pt.gF(A.p,t.d),r=(i-D)/(i+D),.5*(i+D)*(1+(n==68||P==68?0:.2*(1-d.Math.exp(-12*r*r))));},K.yD=function(t,A,n){var P,r,i,D,u,o,Q;return i=Pt.hF(A.p,t.b),D=Pt.hF(A.p,t.d),P=Pt.fF(A.p,t.b),r=Pt.fF(A.p,t.d),u=Pt.jF(A.p,t.b),o=Pt.jF(A.p,t.d),Q=n*n,181.16*i*D*P*r/((d.Math.sqrt(P/u)+d.Math.sqrt(r/o))*Q*Q*Q);},K.zD=function(t,A,n,P){var r,i;this.g=A,this.b=A.a[n],this.d=A.a[P],this.a=n,this.c=P,r=K.xD(this,t),i=K.yD(this,t,r),this.e=Pt.gF(t.p,this.b),this.f=Pt.gF(t.p,this.d),(this.e==68&&this.f==65||this.e==65&&this.f==68)&&(r=r*.8,i*=.5),this.i=r,this.j=i;},K.AD=function(t,A,n,P){var r,i,D,u,o;for(o=new y.Mj(),i=0;i<A.q;i++)for(D=i+1;D<A.q;D++)r=new K.iD(i,D),u=y.xnb(n.a,r)!=null?y.xnb(n.a,r):(K.eD(),K.dD),(u==(K.eD(),K._C)||u==K.dD)&&K.HD(new K.MD(A,i,D))<P&&y.sj(o,new K.zD(t,A,i,D));return o;},N(357,1,Ue,K.zD),U.cc=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b;return P&&(e.Tl(this.g,this.a)==1||e.Tl(this.g,this.c)==1)?0:(o=K.HD(new K.PD(t,this.a,this.c)),Q=o*o,L=Q*Q*Q*o,r=1.07*this.i/(o+s9*this.i),i=r*r,D=i*i*i*r,h=this.i*this.i,b=h*h*h*this.i,u=1.12*b/(L+h9*b)-2,s=D*u*this.j,s);},U.dc=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b;for(r=K.HD(new K.PD(t,this.a,this.c)),D=r/this.i,u=D*D,o=u*u*u,Q=o*D,L=Q+h9,s=1.07/(D+1.07-1),h=s*s,b=h*h*h*s,n=this.j/this.i*b*(T9*o/(L*L)+(T9/L+14)/(D+s9)),i=0;i<3;i++)P=.01*this.i,r>0&&(P=n*(t[3*this.a+i]-t[3*this.c+i])/r),A[3*this.a+i]+=P,A[3*this.c+i]-=P;},U.a=0,U.b=0,U.c=0,U.d=0,U.e=0,U.f=0,U.i=0,U.j=0,l.Z8=O(357),K.DD=function(t,A){return d.Math.acos((t.a*A.a+t.b*A.b+t.c*A.c)/(d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)*d.Math.sqrt(A.a*A.a+A.b*A.b+A.c*A.c)));},K.ED=function(t,A){return new K.KD(t.b*A.c-t.c*A.b,t.c*A.a-t.a*A.c,t.a*A.b-t.b*A.a);},K.FD=function(t,A){return d.Math.sqrt((t.a-A.a)*(t.a-A.a)+(t.b-A.b)*(t.b-A.b)+(t.c-A.c)*(t.c-A.c));},K.GD=function(t,A){return t.a*A.a+t.b*A.b+t.c*A.c;},K.HD=function(t){return d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c);},K.ID=function(t){return d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)>0?new K.KD(t.a/d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),t.b/d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),t.c/d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)):new K.KD(0,0,0);},K.JD=function(t,A){return new K.KD(A.a-t.a,A.b-t.b,A.c-t.c);},K.KD=function(t,A,n){this.a=t,this.b=A,this.c=n;},K.LD=function(t,A){this.a=t.J[A].a,this.b=t.J[A].b,this.c=t.J[A].c;},K.MD=function(t,A,n){K.ND.call(this,K.JD(new K.LD(t,A),new K.LD(t,n)));},K.ND=function(t){this.a=t.a,this.b=t.b,this.c=t.c;},K.OD=function(t,A){this.a=t[3*A],this.b=t[3*A+1],this.c=t[3*A+2];},K.PD=function(t,A,n){K.ND.call(this,K.JD(new K.OD(t,A),new K.OD(t,n)));},N(22,1,{22:1},K.KD,K.LD,K.MD,K.OD,K.PD),U.mb=function(t){var A;return t===this?true:Kt(t,22)?(A=t,d.Math.abs(this.a-A.a)<.01&&d.Math.abs(this.b-A.b)<.01&&d.Math.abs(this.c-A.c)<.01):false;},U.ob=function(){return Rt(this.a)+Rt(this.b)+Rt(this.c);},U.a=0,U.b=0,U.c=0,l.$8=O(22),Pt.SD=function(t,A,n,P,r,i){var D,u,o,Q,L,s,h,b,c;for(c=v(F(l.h7,1),Ft,6,15,[0,0,0]),Q=v(F(l.h7,1),Ft,6,15,[0,0,0]),u=v(F(l.j7,1),H,6,15,[A.C[n],A.C[P],A.C[r]]),o=1.75,L=0;L<3;L++)switch(u[L]){case 1:c[L]=1.395;break;case 6:c[L]=2.494,Q[L]=1.016;break;case 7:c[L]=2.711,Q[L]=1.113;break;case 8:c[L]=3.045,Q[L]=1.337;break;case 9:c[L]=2.847;break;case 14:c[L]=2.35,Q[L]=.811;break;case 15:c[L]=2.35,Q[L]=1.068;break;case 16:c[L]=2.98,Q[L]=1.249;break;case 17:c[L]=2.909,Q[L]=1.078;break;case 35:c[L]=3.017;break;case 53:c[L]=3.086;}return s=Pt.pE(t.a.d,A,n,P),h=Pt.pE(t.a.d,A,P,r),D=(s-h)*(s-h)/((s+h)*(s+h)),b=n9*i,r1.qF(A,n,P,r,4)?o*=.85:r1.qF(A,n,P,r,3)&&(o*=.05),o*c[0]*Q[1]*c[2]/((s+h)*b*b*d.Math.exp(2*D));},Pt.TD=function(t,A,n,P,r){var i;if(r1.qF(A,n,P,r,3))return 60;if(r1.qF(A,n,P,r,4))return 90;switch(i=A.a[P],Pt._D(t.a.b,i)){case 2:if(A.C[P]==8)return 105;if(Pt.bE(t.a.b,i))return 180;break;case 3:if(Pt.fE(t.a.b,i)==3&&Pt.cE(t.a.b,i)==0)return A.C[P]==7?107:92;break;case 4:return 109.45;}return 120;},Pt.UD=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b;for(i=A.a[n],u=A.a[P],o=A.a[r],L=r1.pF(t.a,A,n,P,r),h=-1,s=0;s<5&&h<0;s++)D=t.a.g.a[i-1][s],Q=t.a.g.a[o-1][s],D>Q&&(D=(b=T.Aib(Q),T.Aib(Q=D),b).a),h=K.XC(v(F(l.j7,1),H,6,15,[2,1,3,0]),v(F(l.j7,1),H,6,15,[u,D,Q,L]),t);return h;},Pt.VD=function(t,A,n,P,r){var i;return i=Pt.UD(t,A,n,P,r),i<0?Pt.SD(t,A,n,P,r,Pt.TD(t,A,n,P,r)):d.Math.abs(T.Xhb(t.b[i][4]))<.001?Pt.SD(t,A,n,P,r,T.Xhb(t.b[i][5])):T.Xhb(t.b[i][4]);},Pt.WD=function(t,A,n,P,r){var i;return i=Pt.UD(t,A,n,P,r),i<0?Pt.TD(t,A,n,P,r):T.Xhb(t.b[i][5]);},Pt.XD=function(t){this.b=K.nC("/resources/forcefield/mmff94/angle.csv"),this.a=t;},N(377,1,{},Pt.XD),U.kc=function(t,A){return T.Yhb(this.b[t][A]);},U.lc=function(){return this.b.length;},l._8=O(377),Pt.$D=function(t,A){return t.a[K.VC(0,A,0,t.a.length,true,t)][6]>0;},Pt._D=function(t,A){return t.a[K.VC(0,A,0,t.a.length,true,t)][2];},Pt.aE=function(t){return t.a.length;},Pt.bE=function(t,A){var n;return n=K.VC(0,A,0,t.a.length,true,t),n>=0&&t.a[n][7]>0;},Pt.cE=function(t,A){return t.a[K.VC(0,A,0,t.a.length,true,t)][5];},Pt.dE=function(t,A){return t.a[K.VC(0,A,0,t.a.length,true,t)][4];},Pt.eE=function(t,A){return t.a[K.VC(0,A,0,t.a.length,true,t)][8]>0;},Pt.fE=function(t,A){return t.a[K.VC(0,A,0,t.a.length,true,t)][3];},Pt.gE=function(){this.a=K.oC("/resources/forcefield/mmff94/atom.csv");},N(372,1,{},Pt.gE),U.kc=function(t,A){return this.a[t][A];},U.lc=function(){return Pt.aE(this);},l.a9=O(372),Pt.jE=function(t,A){return T.Xhb(t.a[A][3]);},Pt.kE=function(t,A){return T.Xhb(t.a[A][2]);},Pt.lE=function(){this.a=K.nC("/resources/forcefield/mmff94/bndk.csv");},N(384,1,{},Pt.lE),U.kc=function(t,A){return T.Yhb(this.a[t][A]);},U.lc=function(){return this.a.length;},l.b9=O(384),Pt.oE=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g;return L=r1.AF(t.a,A,e.Xo(A,n,P)),i=A.a[n],u=A.a[P],i>u&&(i=(c=T.Aib(u),T.Aib(u=i),c).a),h=K.XC(v(F(l.j7,1),H,6,15,[1,2,0]),v(F(l.j7,1),H,6,15,[i,u,L]),t),h>=0?T.Xhb(t.b[h][3]):(r=A.C[n],D=A.C[P],r>D&&(r=(b=T.Aib(D),T.Aib(D=r),b).a),g=Pt.pE(t,A,n,P),Q=K.XC(v(F(l.j7,1),H,6,15,[0,1]),v(F(l.j7,1),H,6,15,[r,D]),t.a.c),Q>=0?(s=d.Math.pow(Pt.kE(t.a.c,Q)/g,6),Pt.jE(t.a.c,Q)*s):(o=K.XC(v(F(l.j7,1),H,6,15,[0,1]),v(F(l.j7,1),H,6,15,[K.IC(r),K.IC(D)]),t.a.j),d.Math.pow(10,-(g-Pt.ME(t.a.j,o,2))/Pt.ME(t.a.j,o,3))));},Pt.pE=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;return L=r1.AF(t.a,A,e.Xo(A,n,P)),D=A.a[n],Q=A.a[P],D>Q&&(D=(B=T.Aib(Q),T.Aib(Q=D),B).a),c=K.XC(v(F(l.j7,1),H,6,15,[1,2,0]),v(F(l.j7,1),H,6,15,[D,Q,L]),t),c>=0?T.Xhb(t.b[c][4]):(i=A.C[n],o=A.C[P],i>o&&(i=(g=T.Aib(o),T.Aib(o=i),g).a),r=K.WC(i,t.a.f),u=K.WC(o,t.a.f),k=Pt.CE(t.a.f,r),M=Pt.CE(t.a.f,u),h=Pt.AE(t.a.f,r),b=Pt.AE(t.a.f,u),s=i==1||o==1?.05:.085,k+M-s*d.Math.pow(d.Math.abs(h-b),1.4));},Pt.qE=function(t){this.b=K.nC("/resources/forcefield/mmff94/bond.csv"),this.a=t;},N(380,1,{},Pt.qE),U.kc=function(t,A){return T.Yhb(this.b[t][A]);},U.lc=function(){return this.b.length;},l.c9=O(380),Pt.tE=function(t,A,n,P){return K.VC(1,A,0,n,P,new Pt.xE(t));},Pt.uE=function(t,A){return T.Xhb(t.b[A-1][2]);},Pt.vE=function(t,A,n,P){var r,i,D,u,o,Q,L;return L=n>P?1:-1,Q=n>P?P:n,o=n>P?n:P,i=Pt.tE(t,Q,t.a.length,true),r=Pt.tE(t,Q,t.a.length,false),i==-1||r==-1||(u=K.VC(2,o,i,r+1,true,new Pt.xE(t)),D=K.VC(2,o,i,r+1,false,new Pt.xE(t)),u==-1||D==-1)?T.Xhb(t.b[n-1][1])-T.Xhb(t.b[P-1][1]):A==0&&T.Yhb(t.a[u][0])==0?L*T.Xhb(t.a[u][3]):A==1&&T.Yhb(t.a[D][0])==1?L*T.Xhb(t.a[D][3]):T.Xhb(t.b[n-1][1])-T.Xhb(t.b[P-1][1]);},Pt.wE=function(){this.b=K.nC("/resources/forcefield/mmff94/pbci.csv"),this.a=K.nC("/resources/forcefield/mmff94/bci.csv");},N(378,1,{},Pt.wE),l.e9=O(378),Pt.xE=function(t){this.a=t;},N(196,1,{},Pt.xE),U.kc=function(t,A){return T.Yhb(this.a.a[t][A]);},U.lc=function(){return this.a.a.length;},l.d9=O(196),Pt.AE=function(t,A){return T.Xhb(t.a[A][2]);},Pt.BE=function(t){return t.a.length;},Pt.CE=function(t,A){return T.Xhb(t.a[A][1]);},Pt.DE=function(){this.a=K.nC("/resources/forcefield/mmff94/covrad.csv");},N(385,1,{},Pt.DE),U.kc=function(t,A){return T.Yhb(this.a[t][A]);},U.lc=function(){return Pt.BE(this);},l.f9=O(385),Pt.GE=function(){this.a=K.oC("/resources/forcefield/mmff94/def.csv");},N(387,1,{},Pt.GE),l.g9=O(387),Pt.HE=function(t,A,n,P,r){var i,D,u,o;return i=K.HC(A.C[n]),D=K.HC(A.C[P]),u=K.HC(A.C[r]),i>u&&(i=(o=T.Aib(u),T.Aib(u=i),o).a),K.XC(v(F(l.j7,1),H,6,15,[1,0,2]),v(F(l.j7,1),H,6,15,[D,i,u]),t);},Pt.IE=function(t,A,n,P,r){var i,D,u,o;return i=K.HC(A.C[n]),D=K.HC(A.C[r]),u=i>D?4:3,o=Pt.HE(t,A,n,P,r),o>=0?T.Xhb(t.a[o][u]):0;},Pt.JE=function(){this.a=K.nC("/resources/forcefield/mmff94/dfsb.csv");},N(386,1,{},Pt.JE),U.kc=function(t,A){return T.Yhb(this.a[t][A]);},U.lc=function(){return this.a.length;},l.h9=O(386),Pt.ME=function(t,A,n){return T.Xhb(t.a[A][n]);},Pt.NE=function(){this.a=K.nC("/resources/forcefield/mmff94/herschbachlaurie.csv");},N(388,1,{},Pt.NE),U.kc=function(t,A){return T.Yhb(this.a[t][A]);},U.lc=function(){return this.a.length;},l.i9=O(388),Pt.QE=function(t,A,n,P,r,i){var D,u,o,Q,L,s;for(D=A.a[n],L=v(F(l.j7,1),H,6,15,[A.a[P],A.a[r],A.a[i]]),u=0;u<4;u++){for(s=j(l.j7,H,6,3,15,1),Q=0;Q<3;Q++)s[Q]=t.a.g.a[L[Q]-1][u+1];if(w.vzb(s,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),o=K.XC(v(F(l.j7,1),H,6,15,[1,0,2,3]),v(F(l.j7,1),H,6,15,[D,s[0],s[1],s[2]]),t),o>=0)return T.Xhb(t.b[o][4]);}return 0;},Pt.RE=function(t,A){this.b=K.nC(A),this.a=t;},N(379,1,{},Pt.RE),U.kc=function(t,A){return T.Yhb(this.b[t][A]);},U.lc=function(){return this.b.length;},l.j9=O(379),Pt.UE=function(t,A,n,P,r){var i,D,u,o,Q;return i=A.a[n],D=A.a[P],u=A.a[r],o=r1.oF(t.a,A,n,P,r),i>u&&(i=(Q=T.Aib(u),T.Aib(u=i),Q).a),K.XC(v(F(l.j7,1),H,6,15,[2,1,3,0]),v(F(l.j7,1),H,6,15,[D,i,u,o]),t);},Pt.VE=function(t,A,n,P,r){var i,D,u,o,Q,L;return i=A.a[n],D=A.a[r],o=r1.BF(t.a,A,n,P),Q=r1.BF(t.a,A,P,r),L=Pt.UE(t,A,n,P,r),u=i>D||i==D&&o<Q?1:0,L>=0?T.Xhb(t.b[L][4+u]):Pt.IE(t.a.i,A,n,P,r);},Pt.WE=function(t){this.b=K.nC("/resources/forcefield/mmff94/stbn.csv"),this.a=t;},N(381,1,{},Pt.WE),U.kc=function(t,A){return T.Yhb(this.b[t][A]);},U.lc=function(){return this.b.length;},l.k9=O(381),Pt.ZE=function(t,A,n,P,r,i){var D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt;for(L=A.a[n],s=A.a[P],h=A.a[r],b=A.a[i],ut=r1.EF(t.a,A,n,P,r,i),Dt=ut>10?ut/10|0:ut,lt=ut>10?ut-Dt*10:0,Z=-1,z=0,I=0,tt=0,S=Dt,it=5;(z<it&&(Z==-1||it==4)||z==4&&Dt==5&&lt>0)&&(it==5&&z==4&&(it=4,z=0,S=lt),z==1?(I=1,tt=3):z==2?(I=3,tt=1):(I=z,tt=z),k=t.a.g.a[L-1][d.Math.min(I+1,4)],M=s,p=h,q=t.a.g.a[b-1][d.Math.min(tt+1,4)],M>p?(p=(G=T.Aib(M),T.Aib(M=p),G).a,q=(ot=T.Aib(k),T.Aib(k=q),ot).a):M==p&&k>q&&(q=(ot=T.Aib(k),T.Aib(k=q),ot).a),Z=K.XC(v(F(l.j7,1),H,6,15,[2,3,1,4,0]),v(F(l.j7,1),H,6,15,[M,p,k,q,S]),t),!(Z!=-1&&it==4));)++z;if(Z>=0)return new Pt.dF(t,Z);for(B=e.Xo(A,P,r),u=v(F(l.h7,1),Ft,6,15,[0,0]),o=v(F(l.h7,1),Ft,6,15,[0,0]),Q=v(F(l.h7,1),Ft,6,15,[0,0]),c=v(F(l.j7,1),H,6,15,[A.C[P],A.C[r]]),D=(Pt._D(t.a.b,s)-1)*(Pt._D(t.a.b,h)-1),V=0;V<2;V++)switch(c[V]){case 6:u[V]=2,o[V]=2.12;break;case 7:u[V]=2,o[V]=1.5;break;case 8:u[V]=2,o[V]=.2,Q[V]=2;break;case 14:u[V]=1.25,o[V]=1.22;break;case 15:u[V]=1.25,o[V]=2.4;break;case 16:u[V]=1.25,o[V]=.49,Q[V]=8;}return Pt.bE(t.a.b,s)||Pt.bE(t.a.b,h)?new Pt.cF(0,0):Pt.$D(t.a.b,s)&&Pt.$D(t.a.b,h)&&B<A.g&&e.pt(A.p,B)?(g=Pt.fE(t.a.b,s)==3&&Pt.fE(t.a.b,h)==4||Pt.fE(t.a.b,s)==4&&Pt.fE(t.a.b,h)==3?3:6,Qt=Pt.dE(t.a.b,s)==0&&Pt.dE(t.a.b,h)==0?.5:.3,new Pt.cF(g*Qt*d.Math.sqrt(u[0]*u[1]),0)):e.fm(A,B)==2?(Qt=Pt.cE(t.a.b,s)==2&&Pt.cE(t.a.b,h)==2?1:.4,new Pt.cF(6*Qt*d.Math.sqrt(u[0]*u[1]),0)):Pt._D(t.a.b,s)==4&&Pt._D(t.a.b,h)==4?new Pt.cF(0,d.Math.sqrt(o[0]*o[1])/D):Pt._D(t.a.b,s)==4&&Pt._D(t.a.b,h)!=4?Pt._D(t.a.b,h)==3&&(Pt.fE(t.a.b,h)==4||Pt.fE(t.a.b,h)==34||Pt.cE(t.a.b,h)>0)||Pt._D(t.a.b,h)==2&&(Pt.fE(t.a.b,h)==3||Pt.cE(t.a.b,h)>0)?new Pt.bF():new Pt.cF(0,d.Math.sqrt(o[0]*o[1])/D):Pt._D(t.a.b,h)==4&&Pt._D(t.a.b,s)!=4?Pt._D(t.a.b,s)==3&&(Pt.fE(t.a.b,s)==4||Pt.fE(t.a.b,s)==34||Pt.cE(t.a.b,s)>0)||Pt._D(t.a.b,s)==2&&(Pt.fE(t.a.b,s)==3||Pt.cE(t.a.b,s)>0)?new Pt.bF():new Pt.cF(0,d.Math.sqrt(o[0]*o[1])/D):e.fm(A,B)==1&&Pt.cE(t.a.b,s)>0&&Pt.cE(t.a.b,h)>0||Pt.cE(t.a.b,s)>0&&Pt.dE(t.a.b,h)>0||Pt.dE(t.a.b,s)>0&&Pt.cE(t.a.b,h)>0?Pt.dE(t.a.b,s)>0&&Pt.dE(t.a.b,h)>0?new Pt.bF():Pt.dE(t.a.b,s)>0&&Pt.cE(t.a.b,h)>0?(Qt=0,Pt.cE(t.a.b,s)==1?Qt=.5:K.HC(c[0])==2&&K.HC(c[1])==2?Qt=.3:(K.HC(c[0])!=2||K.HC(c[1])!=2)&&(Qt=.15),new Pt.cF(6*Qt*d.Math.sqrt(u[0]*u[1]),0)):Pt.dE(t.a.b,h)>0&&Pt.cE(t.a.b,s)>0?(Qt=0,Pt.cE(t.a.b,h)==1?Qt=.5:K.HC(c[0])==2&&K.HC(c[1])==2?Qt=.3:(K.HC(c[0])!=2||K.HC(c[1])!=2)&&(Qt=.15),new Pt.cF(6*Qt*d.Math.sqrt(u[0]*u[1]),0)):(Pt.cE(t.a.b,s)==1||Pt.cE(t.a.b,h)==1)&&(c[0]!=6||c[1]!=6)?new Pt.cF(2.4000000000000004*d.Math.sqrt(u[0]*u[1]),0):new Pt.cF(.8999999999999999*d.Math.sqrt(u[0]*u[1]),0):(c[0]==8||c[0]==16)&&(c[1]==8||c[1]==16)?new Pt.cF(-d.Math.sqrt(Q[0]*Q[1]),0):new Pt.cF(0,d.Math.sqrt(o[0]*o[1])/D);},Pt.$E=function(t,A){this.b=K.nC(A),this.a=t;},N(382,1,{},Pt.$E),U.kc=function(t,A){return T.Yhb(this.b[t][A]);},U.lc=function(){return this.b.length;},l.m9=O(382),Pt.bF=function(){this.a=0,this.b=0,this.c=0;},Pt.cF=function(t,A){this.a=0,this.b=t,this.c=A;},Pt.dF=function(t,A){this.a=T.Xhb(t.b[A][5]),this.b=T.Xhb(t.b[A][6]),this.c=T.Xhb(t.b[A][7]);},N(49,1,{},Pt.bF,Pt.cF,Pt.dF),U.pb=function(){return this.a+","+this.b+","+this.c;},U.a=0,U.b=0,U.c=0,l.l9=O(49),Pt.fF=function(t,A){return T.Xhb(t.a[K.VC(0,A,0,t.a.length,true,t)][1]);},Pt.gF=function(t,A){return t.a[K.VC(0,A,0,t.a.length,true,t)][5].a;},Pt.hF=function(t,A){return T.Xhb(t.a[K.VC(0,A,0,t.a.length,true,t)][4]);},Pt.iF=function(t){return t.a.length;},Pt.jF=function(t,A){return T.Xhb(t.a[K.VC(0,A,0,t.a.length,true,t)][2]);},Pt.kF=function(t,A){var n;return n=K.VC(0,A,0,t.a.length,true,t),T.Xhb(t.a[n][3])*d.Math.pow(T.Xhb(t.a[n][1]),.25);},Pt.lF=function(){this.a=K.nC("/resources/forcefield/mmff94/vanderwaals.csv");},N(383,1,{},Pt.lF),U.kc=function(t,A){return T.Yhb(this.a[t][A]);},U.lc=function(){return Pt.iF(this);},l.n9=O(383),r1.oF=function(t,A,n,P,r){var i,D,u,o,Q,L,s;switch(i=A.a[n],D=A.a[r],Q=r1.BF(t,A,n,P),s=r1.BF(t,A,P,r),o=i<=D?Q:s,L=i<D?s:Q,u=r1.pF(t,A,n,P,r),u){case 1:return o>0||o==L?1:2;case 2:return 3;case 3:return 5;case 4:return 4;case 5:return o>0||o==L?6:7;case 6:return 8;case 7:return o>0||o==L?9:10;case 8:return 11;}return 0;},r1.pF=function(t,A,n,P,r){var i,D;return i=r1.BF(t,A,n,P)+r1.BF(t,A,P,r),D=i,r1.qF(A,n,P,r,3)?D+=i>0?4:3:r1.qF(A,n,P,r,4)&&(D+=i>0?6:4),D;},r1.qF=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h;if(h=(e.bw(t,7),t.p),Q=new y.Btb(),y.Atb(Q,T.Aib(A)),y.Atb(Q,T.Aib(n)),y.Atb(Q,T.Aib(P)),e.Xo(t,A,n)>=0&&e.Xo(t,n,P)>=0){for(L=0;L<h.j.a.length;L++)if(y.xj(h.k,L).length==r){for(s=new y.Btb(),D=y.xj(h.j,L),u=0,o=D.length;u<o;++u)i=D[u],y.Atb(s,T.Aib(i));if(y.Ji(s,Q))return  true;}}return  false;},r1.rF=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt,b1,u1,Ht,gt,w1,x1,L1,O1,U1,l1,G1,a1,p0,J1,o1,D0,T0,d0,CA,jA,eA,nA,C0,MA,Z0,M0,YA,Be,mt,Wt,BA,de,me,ke;if(A<t.f&&e.ot(t.p,A)){if(r1.yF(t,A,5)){if(g=new y.Nj(t.q),M=new y.Nj(t.q),c=false,tt=false,ot=false,t.C[A]==6||t.C[A]==7){for(I=0;I<t.e[A];I++)if(mt=t.i[A][I],!!r1.yF(t,mt,5))for(r1.wF(t,A,mt)&&(t.C[mt]==8||t.C[mt]==16||t.C[mt]==7&&t.e[mt]+e.op(t,mt)==3&&!r1.xF(t,mt))&&y.sj(g,T.Aib(mt)),G1=0;G1<t.e[mt];G1++)Wt=t.i[mt][G1],Wt!=A&&r1.yF(t,Wt,5)&&r1.wF(t,A,Wt)&&(t.C[Wt]==8||t.C[Wt]==16||t.C[Wt]==7&&t.e[Wt]+e.op(t,Wt)==3&&!r1.xF(t,Wt))&&y.sj(M,T.Aib(Wt));for(b=new y.Bqb(g);b.a<b.c.a.length;)if(h=y.Aqb(b).a,t.C[h]==8||t.C[h]==16){tt=true;break;}for(k=new y.Bqb(M);k.a<k.c.a.length;)if(B=y.Aqb(k).a,t.C[B]==8||t.C[B]==16){ot=true;break;}for(V=0;V<g.a.length;V++)for(l1=V;l1<M.a.length;l1++)if(r1.wF(t,(w.Pzb(V,g.a.length),g.a[V]).a,(w.Pzb(l1,M.a.length),M.a[l1]).a)){c=true;break;}}switch(t.C[A]){case 6:if(M.a.length==0){for(T0=0,p0=0,o1=0,D0=0,V=0;V<t.e[A];V++)mt=t.i[A][V],t.C[mt]==7&&t.e[mt]+e.op(t,mt)==3&&(++T0,t.s[mt]>0&&!r1.xF(t,mt)&&++p0,r1.yF(t,mt,5)&&++o1,r1.yF(t,mt,6)&&++D0);if((T0==2&&o1>0||T0==3&&o1==2)&&p0>0&&D0==0)return 80;}if(!(g.a.length==0^M.a.length==0)){for(ke=true,me=true,V=0;V<t.e[A];V++)mt=t.i[A][V],(t.C[mt]!=6||!r1.uF(t,mt))&&(ke=false),r1.wF(t,A,mt)&&!(mt<t.f&&e.ot(t.p,mt))&&(me=false);if(g.a.length==0&&M.a.length==0&&!ke&&me||g.a.length!=0&&g.a.length!=0&&(!c||!tt&&!ot))return 78;}if(g.a.length!=0&&(M.a.length==0||tt))return 63;if(M.a.length!=0&&(g.a.length==0||ot))return 64;break;case 7:if(r1.xF(t,A))return 82;if(g.a.length==0&&M.a.length==0)return t.e[A]+e.op(t,A)==3?39:76;if(t.e[A]+e.op(t,A)==3&&g.a.length==0^M.a.length==0)return 81;if(g.a.length!=0&&(M.a.length==0||tt))return 65;if(M.a.length!=0&&(g.a.length==0||ot))return 66;if(g.a.length!=0&&M.a.length!=0)return 79;break;case 8:return 59;case 16:return 44;}}if(r1.yF(t,A,6))switch(t.C[A]){case 6:return 37;case 7:return r1.xF(t,A)?69:t.e[A]+e.op(t,A)==3?58:38;}}switch(t.C[A]){case 3:if(t.j[A]==0)return 92;break;case 6:if(t.e[A]+e.op(t,A)==4)return (t.p&&A<t.f?e.dt(t.p,A):0)==3?22:(t.p&&A<t.f?e.dt(t.p,A):0)==4?20:1;if(t.e[A]+e.op(t,A)==3){for(d0=0,CA=0,eA=0,Z0=0,q=0,I=0;I<t.e[A];I++)mt=t.i[A][I],e.fm(t,e.Xo(t,A,mt))==2&&(q=t.C[mt]),t.e[mt]+e.op(t,mt)==1?(t.C[mt]==8&&++eA,t.C[mt]==16&&++Z0):t.C[mt]==7&&(t.e[mt]+e.op(t,mt)==3?++CA:t.e[mt]+e.op(t,mt)==2&&e.fm(t,e.Xo(t,A,mt))==2&&++d0);return CA>=2&&d0==0&&q==7?57:eA==2||Z0==2?41:(t.p&&A<t.f?e.dt(t.p,A):0)==4&&q==6?30:q==7||q==8||q==15||q==16?3:2;}if(t.e[A]+e.op(t,A)==2)return 4;if(t.e[A]+e.op(t,A)==1)return 60;break;case 7:for(YA=0,L1=false,Z=0;Z<t.e[A];Z++)if(mt=t.i[A][Z],t.C[mt]==8&&t.e[mt]+e.op(t,mt)==1&&++YA,e.vp(t,A)+e.op(t,A)>=3&&(t.C[mt]==15||t.C[mt]==16)){for(MA=0,l1=0;l1<t.e[mt];l1++)Wt=t.i[mt][l1],t.C[Wt]==8&&t.e[Wt]+e.op(t,Wt)==1&&++MA;L1||(L1=MA>=2);}if(t.e[A]+e.op(t,A)==4)return r1.xF(t,A)?68:34;if(t.e[A]+e.op(t,A)==3){if(e.vp(t,A)+e.op(t,A)>=4){for(S=false,V=0;V<t.e[A];V++)if(mt=t.i[A][V],e.fm(t,e.Xo(t,A,mt))==2&&(S=t.C[mt]==7||t.C[mt]==6,t.C[mt]==6))for(l1=0;S&&l1<t.e[mt];l1++)Wt=t.i[mt][l1],Wt!=A&&(S=!(t.C[Wt]==7&&t.e[Wt]+e.op(t,Wt)==3));if(YA==1)return 67;if(YA>=2)return 45;if(S)return 54;}if(e.vp(t,A)+e.op(t,A)>=3){for(yt=false,ct=false,wt=false,bt=false,Nt=false,Zt=false,n=0,P=0,r=0,i=0,u=0,L=0,V=0;V<t.e[A];V++)if(mt=t.i[A][V],t.C[mt]==6){for(Nt=true,mt<t.f&&e.ot(t.p,mt)&&(t.p&&mt<t.f?e.dt(t.p,mt):0)==6&&(Zt=true),r=0,i=0,u=0,L=0,p0=0,z=0,l1=0;l1<t.e[mt];l1++){if(Wt=t.i[mt][l1],p=e.Xo(t,mt,Wt),e.fm(t,p)==2&&(t.C[Wt]==8||t.C[Wt]==16)&&(yt=true),(e.fm(t,p)==2||p<t.g&&e.pt(t.p,p)&&(t.C[Wt]==6||t.C[Wt]==7&&r1.vF(t,Wt)==1))&&(n=t.C[Wt]),e.fm(t,p)==3&&(P=t.C[Wt]),t.C[Wt]==7&&t.e[Wt]+e.op(t,Wt)==3){for(t.s[Wt]==1&&++p0,r1.yF(t,mt,6)&&++z,Q=0,a1=0;a1<t.e[Wt];a1++)BA=t.i[Wt][a1],t.C[BA]==8&&++Q;Q<2&&++i;}t.C[Wt]==7&&t.e[Wt]+e.op(t,Wt)==2&&(e.fm(t,p)==2||p<t.g&&e.pt(t.p,p))&&++r,Wt<t.f&&e.ot(t.p,Wt)&&(t.C[Wt]==8&&++u,t.C[Wt]==16&&++L);}n==7&&(i==2&&r==0&&p0>0&&z==0&&t.e[mt]+e.op(t,mt)<4&&(ct=true),i==3&&(wt=true));}else if(t.C[mt]==7){for(D=0,o=0,s=0,l1=0;l1<t.e[mt];l1++)if(Wt=t.i[mt][l1],p=e.Xo(t,mt,Wt),e.fm(t,p)==2){if(t.C[Wt]==6){for(a1=0;a1<t.e[Wt];a1++)BA=t.i[Wt][a1],BA!=A&&(t.C[BA]==7?++D:t.C[BA]==8?++o:t.C[BA]==16&&++s);D==0&&o==0&&s==0&&!Zt&&(bt=true);}t.C[Wt]==7&&!Zt&&(bt=true);}}if(Nt){if(P==7&&(L1=true),ct)return 55;if(wt)return 56;if(!yt&&!L1&&(u==0&&L==0&&Zt||n==6||n==7||n==15||P==6))return 40;}if(!L1&&(yt||bt))return 10;}}if(t.e[A]+e.op(t,A)==2){if(e.vp(t,A)+e.op(t,A)==4){for(Dt=false,V=0;!Dt&&V<t.e[A];V++)mt=t.i[A][V],Dt=e.fm(t,e.Xo(t,A,mt))==3;return Dt?61:53;}if(e.vp(t,A)+e.op(t,A)==3){for(Bt=false,ut=false,V=0;V<t.e[A];V++)mt=t.i[A][V],e.fm(t,e.Xo(t,A,mt))==2&&(Bt=t.C[mt]==8&&YA==1,ut=t.C[mt]==6||t.C[mt]==7);if(Bt&&!ut)return 46;if(ut)return 9;}if(e.vp(t,A)+e.op(t,A)>=2){for(Et=false,V=0;V<t.e[A];V++)if(mt=t.i[A][V],t.C[mt]==16){for(Be=0,l1=0;l1<t.e[mt];l1++)Wt=t.i[mt][l1],t.C[Wt]==8&&t.e[Wt]+e.op(t,Wt)==1&&++Be;Et=Be==1;}if(Et)return 48;if(!L1)return 62;}}if(L1)return 43;if(t.e[A]+e.op(t,A)==1){for(qt=false,lt=false,V=0;!qt&&!lt&&V<t.e[A];V++)if(mt=t.i[A][V],qt=e.fm(t,e.Xo(t,A,mt))==3,t.C[mt]==7&&t.e[mt]+e.op(t,mt)==2)for(l1=0;!lt&&l1<t.e[mt];l1++)Wt=t.i[mt][l1],lt=t.C[Wt]==7&&t.e[Wt]+e.op(t,Wt)==2||t.C[Wt]==6&&t.e[Wt]+e.op(t,Wt)==3;if(qt)return 42;if(lt)return 47;}return 8;case 8:if(t.e[A]+e.op(t,A)==3)return 49;if(t.e[A]+e.op(t,A)==2){if(e.vp(t,A)+e.op(t,A)==3)return 51;for(J1=0,V=0;V<t.e[A];V++)mt=t.i[A][V],t.C[mt]==1&&++J1;return J1+e.op(t,A)==2?70:6;}if(t.j[A]<=1){for(jA=0,C0=0,M0=0,Ht=t.e[A]-t.j[A]+e.op(t,A)>0,it=false,G=false,u1=false,jt=false,gt=false,xt=false,It=false,U1=false,x1=false,O1=false,w1=false,V=0;V<t.e[A]&&!(Ht||it||G||u1||jt||gt||xt||It||U1||x1||O1||w1);V++){if(mt=t.i[A][V],t.C[mt]==6||t.C[mt]==7||t.C[mt]==16)for(l1=0;l1<t.e[mt];l1++)Wt=t.i[mt][l1],t.C[Wt]==7&&t.e[Wt]+e.op(t,Wt)==2&&++jA,t.C[Wt]==8&&t.e[Wt]+e.op(t,Wt)==1&&++C0,t.C[Wt]==16&&t.e[Wt]+e.op(t,Wt)==1&&++M0;Ht=t.C[mt]==1,t.C[mt]==6&&(it=C0==2,G=e.fm(t,e.Xo(t,A,mt))==2,u1=e.fm(t,e.Xo(t,A,mt))==1&&C0==1),t.C[mt]==7&&(jt=e.fm(t,e.Xo(t,A,mt))==2,e.fm(t,e.Xo(t,A,mt))==1&&C0==1&&(gt=t.e[mt]+e.op(t,mt)==2||e.vp(t,mt)+e.op(t,mt)==3,xt=e.vp(t,mt)+e.op(t,mt)==4),It=C0>=2),t.C[mt]==16&&(U1=M0==1,x1=e.fm(t,e.Xo(t,A,mt))==1||e.fm(t,e.Xo(t,A,mt))==2&&C0+jA>1,O1=e.fm(t,e.Xo(t,A,mt))==2&&C0+jA==1),w1=t.C[mt]==15||t.C[mt]==17;}if(u1||gt||Ht)return 35;if(it||It||xt||U1||x1||w1)return 32;if(G||jt||O1)return 7;}break;case 9:if(t.j[A]==1)return 11;if(t.j[A]==0)return 89;break;case 11:if(t.j[A]==0)return 93;break;case 12:if(t.j[A]==0)return 99;break;case 14:return 19;case 15:if(t.e[A]+e.op(t,A)==4)return 25;if(t.e[A]+e.op(t,A)==3)return 26;if(t.e[A]+e.op(t,A)==2)return 75;break;case 16:if(t.e[A]+e.op(t,A)==3||t.e[A]+e.op(t,A)==4){for(nA=0,M0=0,Qt=false,V=0;V<t.e[A];V++)mt=t.i[A][V],t.C[mt]==6&&e.fm(t,e.Xo(t,A,mt))==2&&(Qt=true),(t.j[mt]==1&&t.C[mt]==8||t.e[mt]+e.op(t,mt)==2&&t.C[mt]==7)&&++nA,t.j[mt]==1&&t.C[mt]==16&&++M0;return t.e[A]+e.op(t,A)==3&&nA==2&&Qt||t.e[A]+e.op(t,A)==4?18:nA>0&&M0>0||nA==2&&!Qt?73:17;}if(t.e[A]+e.op(t,A)==2){for(b1=false,V=0;V<t.e[A];V++)mt=t.i[A][V],t.C[mt]==8&&e.fm(t,e.Xo(t,A,mt))==2&&(b1=true);return b1?74:15;}if(t.e[A]+e.op(t,A)==1){for(de=0,Qt=false,V=0;V<t.e[A];V++){for(mt=t.i[A][V],l1=0;l1<t.e[mt];l1++)Wt=t.i[mt][l1],t.C[Wt]==16&&t.e[Wt]+e.op(t,Wt)==1&&++de;t.C[mt]==6&&e.fm(t,e.Xo(t,A,mt))==2&&(Qt=true);}return Qt&&de!=2?16:72;}break;case 17:if(t.j[A]==4){for(eA=0,V=0;V<t.e[A];V++)mt=t.i[A][V],t.C[mt]==8&&++eA;if(eA==4)return 77;}if(t.j[A]==1)return 12;if(t.j[A]==0)return 90;break;case 19:if(t.j[A]==0)return 94;break;case 20:if(t.j[A]==0)return 96;break;case 26:if(t.j[A]==0){if(t.s[A]==2)return 87;if(t.s[A]==3)return 88;}break;case 29:if(t.j[A]==0){if(t.s[A]==1)return 97;if(t.s[A]==2)return 98;}break;case 30:if(t.j[A]==0)return 95;break;case 35:if(t.j[A]==1)return 13;if(t.j[A]==0)return 91;break;case 53:if(t.j[A]==1)return 14;}return 0;},r1.sF=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;for(P=0;P<t.e[A];P++)switch(L=t.i[A][P],t.C[L]){case 6:return 5;case 7:switch(r1.rF(t,L)){case 8:case 39:case 62:case 67:case 68:return 23;case 34:case 54:case 55:case 56:case 58:case 81:return 36;case 9:return 27;default:return 28;}case 8:switch(r1.rF(t,L)){case 49:return 50;case 51:return 52;case 70:return 31;case 6:for(r=false,i=false,D=false,u=false,o=0;o<t.e[L];o++){if(s=t.i[L][o],t.C[s]==6)for(Q=0;Q<t.e[s];Q++)h=t.i[s][Q],h!=L&&(n=e.Xo(t,s,h),(t.C[h]==6||t.C[h]==7)&&(e.fm(t,n)==2||n<t.g&&e.pt(t.p,n))&&(r=true),t.C[h]==8&&e.fm(t,n)==2&&(i=true));t.C[s]==15&&(D=true),t.C[s]==16&&(u=true);}if(i||D)return 24;if(r)return 29;if(u)return 33;default:return 21;}case 14:return 5;case 16:case 15:return 71;}return 0;},r1.tF=function(t,A){return t.a[A]>-1?t.a[A]:t.C[A]==1?r1.sF(t,A):r1.rF(t,A);},r1.uF=function(t,A){var n,P;if((t.u[A]&V0)!=0){for(P=(e.bw(t,7),t.p),n=0;n<P.j.a.length;n++)if(y.xj(P.k,n).length==6&&e.qt(P,n,A))return  true;}return  false;},r1.vF=function(t,A){var n,P,r;for(r=(e.bw(t,7),t.p),P=0,n=0;n<r.j.a.length;n++)e.ct(r,n,A)>=0&&++P;return P;},r1.wF=function(t,A,n){var P,r;if((t.u[A]&V0)==0||(t.u[n]&V0)==0)return  false;for(r=(e.bw(t,7),t.p),P=0;P<r.j.a.length;P++)if(y.xj(r.k,P).length==5&&e.qt(r,P,A)&&e.qt(r,P,n))return  true;return  false;},r1.xF=function(t,A){var n,P;if(t.C[A]==7&&t.e[A]+e.op(t,A)>=3){for(n=0;n<t.e[A];n++)if(P=t.i[A][n],t.C[P]==8&&t.e[P]+e.op(t,P)==1)return  true;}return  false;},r1.yF=function(t,A,n){var P,r;if(A<t.f&&e.ot(t.p,A)){for(r=(e.bw(t,7),t.p),P=0;P<r.j.a.length;P++)if(!(y.xj(r.k,P).length!=n||!e.qt(r,P,A))&&t.b[P]==0)return  true;}return  false;},r1.zF=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;if(k=(e.bw(t,7),t.p),!k.e[A])return 1;if(y.xj(k.k,A).length==6){for(P=y.xj(k.j,A),r=0,i=P.length;r<i;++r)if(n=P[r],e.vp(t,n)+e.op(t,n)!=t.e[n]+e.op(t,n)+1)return 1;for(o=y.xj(k.k,A),Q=0,L=o.length;Q<L;++Q)if(u=o[Q],s=v(F(l.j7,1),H,6,15,[-1,-1]),e.fm(t,u)==1){for(b=0;b<=1;b++)for(D=t.D[b][u],h=0;h<t.e[D];h++)if(c=t.i[D][h],!e.qt(k,A,c)&&e.fm(t,e.Xo(t,D,c))==2){s[b]=c;break;}if(s[0]>-1&&s[1]>-1)for(B=0;B<k.j.a.length;B++){if(e.qt(k,B,s[0])&&e.qt(k,B,s[1])&&t.b[B]==2)return 2;if(e.qt(k,B,s[0])&&e.qt(k,B,s[1])&&t.b[B]!=0)return 1;}}}if(y.xj(k.k,A).length==5)for(g=1,P=y.xj(k.j,A),r=0,i=P.length;r<i;++r){if(n=P[r],e.vp(t,n)+e.op(t,n)==t.e[n]+e.op(t,n)&&g>0){--g;continue;}if(e.vp(t,n)+e.op(t,n)!=t.e[n]+e.op(t,n)+1)return 1;}return 0;},r1.AF=function(t,A,n){return r1.BF(t,A,A.D[0][n],A.D[1][n]);},r1.BF=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b;for(r=A.a[n],i=A.a[P],L=e.Xo(A,n,P),s=true,b=(e.bw(A,7),A.p),h=0;h<b.j.a.length&&s;h++)for(u=y.xj(b.k,h),o=0,Q=u.length;o<Q;++o)if(D=u[o],D==L&&A.b[h]==0){s=false;break;}return e.fm(A,L)==1&&s&&(Pt.$D(t.b,r)&&Pt.$D(t.b,i)||Pt.eE(t.b,r)&&Pt.eE(t.b,i))?1:0;},r1.CF=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;for(u=r1.DF(A),D=j(l.h7,Ft,6,u.length,15,1),P=0;P<u.length;P++){if(B=A.a[P],b=u[P],k=Pt.uE(t.e,B),n=Pt._D(t.b,B),c=0,g=0,d.Math.abs(k)<wn)for(Q=0;Q<A.e[P];Q++)s=A.i[P][Q],u[s]<0&&(b+=u[s]/(2*A.e[s]));if(B==62)for(L=0;L<A.e[P];L++)s=A.i[P][L],u[s]>0&&(b-=u[s]/2);for(o=0;o<A.e[P];o++)s=A.i[P][o],r=e.Xo(A,P,s),h=A.a[s],i=r1.BF(t,A,A.D[0][r],A.D[1][r]),g+=Pt.vE(t.e,i,B,h),c+=u[s];D[P]=(1-n*k)*b+k*c+g;}return D;},r1.DF=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z;D=j(l.h7,Ft,6,t.q,15,1);t:for(i=0;i<D.length;i++){switch(z=t.a[i],D[i]=0,z){case 32:case 72:for(Q=0;Q<t.e[i];Q++){for(k=t.i[i][Q],q=t.a[k],g=0,B=0,s=0;s<t.e[k];s++)M=t.i[k][s],t.C[M]==7&&t.e[M]+e.op(t,M)==2&&!(M<t.f&&e.ot(t.p,M))&&++g,(t.C[M]==8||t.C[M]==16)&&t.e[M]+e.op(t,M)==1&&++B;if(t.C[k]==16&&B==2&&g==1&&(g=0),t.C[k]==6&&B>0){D[i]=B==1?-1:-(B-1)/B;continue t;}if(q==45&&B==3){D[i]=-0.3333333333333333;continue t;}if(q==25&&B>0){D[i]=B==1?0:-(B-1)/B;continue t;}if(q==18&&B>0){D[i]=g+B==2?0:-(g+B-2)/B;continue t;}if(q==73&&B>0){D[i]=B==1?0:-(B-1)/B;continue t;}if(q==77&&B>0){D[i]=-1/B;continue t;}}break;case 76:for(I=0,Z=(e.bw(t,7),t.p),V=0;V<Z.j.a.length;V++)if(e.qt(Z,V,i)){I=V;break;}if(I<Z.j.a.length){for(c=0,n=y.xj(Z.j,I),P=0,r=n.length;P<r;++P)A=n[P],A>-1&&t.a[A]==76&&++c;if(c>0){D[i]=-1/c;continue t;}}break;case 55:case 56:case 81:for(D[i]=t.s[i],b=1,S=0,u=j(l.Feb,ht,6,t.q,16,1),u[i]=true;b>S;)for(S=b,L=0;L<t.q;L++)if(u[L]){for(s=0;s<t.e[L];s++)if(k=t.i[L][s],q=t.a[k],!(q!=57&&q!=80))for(h=0;h<t.e[k];h++)M=t.i[k][h],p=t.a[M],!(p!=55&&p!=56&&p!=81)&&(u[M]||(u[M]=true,D[i]+=t.s[M],++b));}D[i]/=b;continue t;case 61:for(o=0;o<t.e[i];o++)if(k=t.i[i][o],t.a[k]==42){D[i]=1;continue t;}continue t;case 34:case 49:case 51:case 54:case 58:case 92:case 93:case 94:case 97:D[i]=1;continue t;case 87:case 95:case 96:case 98:case 99:D[i]=2;continue t;case 88:D[i]=3;continue t;case 35:case 62:case 89:case 90:case 91:D[i]=-1;continue t;}D[i]=0;}return D;},r1.EF=function(t,A,n,P,r,i){var D,u,o,Q,L,s;return Q=e.Xo(A,P,r),D=r1.BF(t,A,n,P),u=r1.BF(t,A,P,r),o=r1.BF(t,A,r,i),s=u,u==0&&e.fm(A,Q)==1&&(D==1||o==1)&&(s=2),L=r1.FF(A,n,P,r,i),L==4&&e.Xo(A,n,r)==-1&&e.Xo(A,P,i)==-1?40+s:L==5&&(A.a[n]==1||A.a[P]==1||A.a[r]==1||A.a[i]==1)?50+s:s;},r1.FF=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h;if(e.Xo(t,A,n)==-1||e.Xo(t,n,P)==-1||e.Xo(t,P,r)==-1)return 0;if(e.Xo(t,r,A)>=0)return 4;for(s=(e.bw(t,7),t.p),h=new y.Btb(),y.Atb(h,T.Aib(A)),y.Atb(h,T.Aib(n)),y.Atb(h,T.Aib(P)),y.Atb(h,T.Aib(r)),Q=0;Q<s.j.a.length;Q++)if(y.xj(s.k,Q).length==5){for(L=new y.Btb(),D=y.xj(s.j,Q),u=0,o=D.length;u<o;++u)i=D[u],y.Atb(L,T.Aib(i));if(y.Ji(L,h))return 5;}return 0;},N(415,1,{}),l.o9=O(415),e1.HF=function(){e1.HF=D1,e1.GF=(T.Zjb(),k1);},e1.IF=function(t){},e1.JF=function(t){var A,n,P;for(A=0,n=0,P=0;P<t.g.a.length+t.f.a.length;P++)A+=e.Ul(P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length))*(P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length)).r,n+=(P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length)).r;return n!=0?n/A:1;},e1.KF=function(t,A){e1.HF();var n,P,r,i;e1.IF(this),P=new At.AH(t);try{for(i=new Ct.Dgb(),Ct.Cgb(i,"$RXN"+e1.GF),T.Ljb(i.a,A??""),Ct.Cgb(i,e1.GF+(""+e1.GF)),Ct.Cgb(i,hP+At.VH(t,!0,27)),Ct.Cgb(i,e1.GF),Ct.Cgb(i,"  "+P.g.a.length+"  "+P.f.a.length+e1.GF),r=e1.JF(P),n=0;n<P.g.a.length+P.f.a.length;n++)Ct.Cgb(i,"$MOL"+e1.GF),e.$r(new e.bs(n<P.g.a.length?y.xj(P.g,n):y.xj(P.f,n-P.g.a.length),r,null),i);this.a=i.a;}catch(D){if(D=M1(D),Kt(D,19))T.Zjb();else throw Ut(D);}},N(279,1,{},e1.KF),U.a=null,l.p9=O(279),e1.LF=function(t){var A,n;return A=new At.zH(),n=new Ct.ogb(new Ct.zgb(t)),e1.MF(A,n),A;},e1.MF=function(t,A){var n,P;if(P=Ct.ngb(A),n=false,P==null||!T.kjb(w.aAb(P).substr(0,4),"$RXN"))throw Ut(new T.cC("'$RXN' tag not found"));return T.kjb(P,b9)?n=e1.OF(t,A,false):n=e1.NF(t,A,false),n;},e1.NF=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;if(Q=T.Cjb(Ct.ngb(A)),w.aAb(Q).length==0||(t.e=Q),Ct.ngb(A),P=Ct.ngb(A),!n&&T.kjb(w.aAb(P).substr(0,13),hP)&&(r=(w.Xzb(13,w.aAb(P).length+1),w.aAb(P).substr(13)),At.UH(r,true,t)))return  true;for(c=Ct.ngb(A),b=T.Whb(T.Cjb((w.Wzb(0,3,w.aAb(c).length),w.aAb(c).substr(0,3))),P1,St),s=T.Whb(T.Cjb((w.Wzb(3,6,w.aAb(c).length),w.aAb(c).substr(3,3))),P1,St),u=new e.Cs(),D=0;D<b;D++){if(c=Ct.ngb(A),c==null||!T.kjb(w.aAb(c).substr(0,4),"$MOL"))throw Ut(new T.cC(c9));h=new e.tw(),o=new T.Njb();do c=Ct.ngb(A),o.a+=""+c,o.a+=k1;while(c!=null&&!T.kjb(w.aAb(c).substr(0,6),FA));if(c==null)throw Ut(new T.cC(w9));e.xs(u,h,new Ct.ogb(new Ct.zgb(o.a))),y.sj(t.g,h),t.d=-1;}for(i=0;i<s;i++){if(c=Ct.ngb(A),c==null||!T.kjb(w.aAb(c).substr(0,4),"$MOL"))throw Ut(new T.cC(c9));L=new e.tw(),o=new T.Njb();do c=Ct.ngb(A),o.a+=""+c,o.a+=k1;while(c!=null&&!T.kjb(w.aAb(c).substr(0,6),FA));if(c==null)throw Ut(new T.cC(w9));e.xs(u,L,new Ct.ogb(new Ct.zgb(o.a))),y.sj(t.f,L),t.d=-1;}return  true;},e1.OF=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;if(Q=T.Cjb(Ct.ngb(A)),w.aAb(Q).length==0||(t.e=Q),Ct.ngb(A),P=Ct.ngb(A),!n&&T.kjb(w.aAb(P).substr(0,13),hP)&&(r=(w.Xzb(13,w.aAb(P).length+1),w.aAb(P).substr(13)),At.UH(r,true,t)))return  true;if(c=Ct.ngb(A),D=new e.Cs(),c!=null&&T.kjb(w.aAb(c).substr(0,13),"M  V30 COUNTS")){if(b=T.Cjb((w.Xzb(13,w.aAb(c).length+1),w.aAb(c).substr(13))),L=T.yjb(b," "),h=T.Whb(L[0],P1,St),s=T.Whb(L[1],P1,St),h>0){if(c=Ct.ngb(A),T.kjb(g9,c))for(i=0;i<h;i++){u=new e.tw(),o=new T.Njb(),o.a+=j9;do c=Ct.ngb(A),o.a+=""+c,o.a+=k1;while(c!=null&&!T.kjb(w.aAb(c).substr(0,15),Lr));e.xs(D,u,new Ct.ogb(new Ct.zgb(o.a))),y.sj(t.g,u),t.d=-1;}c=Ct.ngb(A);}if(s>0&&(c=Ct.ngb(A),T.kjb(B9,c))){for(i=0;i<s;i++){u=new e.tw(),o=new T.Njb(),o.a+=j9;do c=Ct.ngb(A),o.a+=""+c,o.a+=k1;while(c!=null&&!T.kjb(w.aAb(c).substr(0,15),Lr));e.xs(D,u,new Ct.ogb(new Ct.zgb(o.a))),y.sj(t.f,u),t.d=-1;}c=Ct.ngb(A);}return  true;}return  false;},e1.QF=function(){e1.QF=D1,e1.PF=(T.Zjb(),k1);},e1.RF=function(t){},e1.SF=function(t){var A,n,P;for(A=0,n=0,P=0;P<t.g.a.length+t.f.a.length;P++)A+=e.Ul(P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length))*(P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length)).r,n+=(P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length)).r;return n!=0?n/A:1;},e1.TF=function(t,A){e1.QF();var n,P,r,i,D,u;e1.RF(this),i=new At.AH(t);try{if(u=new Ct.Dgb(),Ct.Cgb(u,b9+e1.PF),T.Ljb(u.a,A??""),Ct.Cgb(u,e1.PF+(""+e1.PF)),Ct.Cgb(u,hP+At.VH(t,!0,11)),Ct.Cgb(u,e1.PF),r=i.g.a.length,P=i.f.a.length,Ct.Cgb(u,K8+r+" "+P+e1.PF),D=e1.SF(i),r>0){for(Ct.Cgb(u,g9+e1.PF),n=0;n<i.g.a.length;n++)Ct.Cgb(u,e.Os(y.xj(i.g,n),D));Ct.Cgb(u,"M  V30 END REACTANT"+e1.PF);}if(P>0){for(Ct.Cgb(u,B9+e1.PF),n=0;n<i.f.a.length;n++)Ct.Cgb(u,e.Os(y.xj(i.f,n),D));Ct.Cgb(u,"M  V30 END PRODUCT"+e1.PF);}Ct.Cgb(u,FA+e1.PF),this.a=u.a;}catch(o){if(o=M1(o),Kt(o,19))T.Zjb();else throw Ut(o);}},N(280,1,{},e1.TF),U.a=null,l.q9=O(280),e1.VF=function(){e1.VF=D1,e1.UF=v(F(l.Ybb,1),st,2,6,["Idorsia No","Actelion No","ID","IDNUMBER","COMPOUND_ID","NAME","COMPND"]);},e1.WF=function(t){var A,n,P,r,i,D,u,o,Q;if(!t.i)return  false;T.Ggb(t.g,0),T.Ggb(t.b,0),t.f=null,Q=false,n=-1,P=null,t.c=t.d==null?null:j(l.Ybb,st,2,t.d.length,6,1),t.e=-1;do{try{if(o=Ct.ngb(t.i),o==null)return T.Ggb(t.g,0),!1;}catch(L){if(L=M1(L),Kt(L,75))return T.Ggb(t.g,0),false;throw Ut(L);}if(Q)T.Sjb(t.b,o),T.Ojb(t.b,10);else if(T.kjb(w.aAb(o).substr(0,1),">"))Q=true,T.Sjb(t.g,FA),T.Ojb(t.g,10),T.Sjb(t.b,o),T.Ojb(t.b,10);else {T.Sjb(t.g,o),T.Ojb(t.g,10),T.kjb(w.aAb(o).substr(0,6),FA)&&(Q=true);continue;}if(t.d!=null)if(w.aAb(o).length==0)n=-1;else if(n==-1){if(P=e1.YF(o),P!=null){for(n=-1,A=0;A<t.d.length;A++)if(T.kjb(P,t.d[A])){n=A;break;}if(t.e==-1){for(i=e1.UF,D=0,u=i.length;D<u;++D)if(r=i[D],T.kjb(P,r)){t.e=n;break;}}}}else t.c[n]==null?t.c[n]=o:t.c[n]=T.gjb(T.gjb(t.c[n],k1),o);}while(!T.kjb(w.aAb(o).substr(0,4),Qn));return  true;},e1.XF=function(t,A){var n,P,r,i;for(i=0,P=new e.rx();i<A;){try{r=Ct.ngb(t.i);}catch(D){if(D=M1(D),Kt(D,75))break;throw Ut(D);}if(r==null)break;T.kjb(w.aAb(r).substr(0,4),Qn)&&++i,T.kjb(w.aAb(r).substr(0,1),">")&&(n=e1.YF(r),n!=null&&e.qx(P,n));}t.d=y.Lj(P.b,j(l.Ybb,st,2,0,6,1));},e1.YF=function(t){var A,n,P,r;if(w.aAb(t).length==0||(w.Xzb(0,w.aAb(t).length),w.aAb(t).charCodeAt(0)!=62))return null;for(P=1,r=0,A=0;P<w.aAb(t).length;){if(w.Xzb(P,w.aAb(t).length),w.aAb(t).charCodeAt(P)==60){if(r!=0)return null;r=P;}else if(w.Xzb(P,w.aAb(t).length),w.aAb(t).charCodeAt(P)==62){if(A!=0)return null;A=P;}++P;}if(r!=0&&r<A)return w.Wzb(r+1,A,w.aAb(t).length),w.aAb(t).substr(r+1,A-(r+1));if(P=w.aAb(t).indexOf("DT",1),P==-1)return null;for(n=P+2;w.aAb(t).length>n&&AA((w.Xzb(n,w.aAb(t).length),w.aAb(t).charCodeAt(n)));)++n;return n==P+2?null:(w.Wzb(P,n,w.aAb(t).length),w.aAb(t).substr(P,n-P));},e1.ZF=function(t,A){return t.c==null?null:t.c[A];},e1.$F=function(t){return t.d==null&&e1.XF(t,10240),t.d;},e1._F=function(t,A){return t.d==null&&e1.XF(t,A),t.d;},e1.aG=function(t){var A,n;return t.f||(n=new e.Cs(),t.a&&(n.a=true),t.f=e.fs(n,(A=t.g.a,A)),t.f&&(t.f.P==null||w.aAb(t.f.P).length==0)&&e.Un(t.f,t.e!=-1&&t.c!=null?t.c[t.e]:t.f?t.f.P:null)),t.f;},e1.bG=function(t,A){e1.VF(),this.d=A,this.i=new Ct.ogb(t),this.g=new T.Vjb(),this.b=new T.Vjb();},N(281,415,{},e1.bG),U.a=false,U.e=0,l.r9=O(281),nt.fG=function(){nt.fG=D1,nt.cG=v(F(l.k7,1),e0,6,14,[524290,524292,589826,589827,589828,589829,589832,589833,589837,589838,598019,598020,598024,606211,688130,688131,688132,688137,688142,696324,819202,819203,819204,819205,819214,827396,1048580,1114114,1114115,1114116,1114120,1212420,1220611,1343492,1343496,1638402,1638403,17367042,17367043,17367044,17367045,17367048,17367049,17367053,17367054,17375236,17465346,17465348,17465353,17473540,1208483842,1208483848,1208483912,1208549378,1208549379,1208549380,1208549384,1208549388,1208549442,1208549443,1208549444,1208549448,1208557571,1208565763,1208565827,1409875970,1409875971,1409875972,1409875976,1409876034,1409876035,1409876036,1409876040,1409884163,1409974274,1409974280,1409974338,1678245892,1678311426,1678311427,1678311428,1678311432,1678311490,1678311491,1678311492,1678311496,1678409730,1678409731,1678409732,1678409794,1678540802,1678540808,1678540810,1678540866,2282291202,2282291203,2282291266,2282291267,2282389506,2282389507,2282389570,2282389571,2282389572,2282520578,2282520579,2282520642,2282520643,2282537027,2483617794,2483617795,2483617858,2483617859,2483617860,2483617864,2483716098,2483716099,2483716162,2483716163,2484150275,2484248579,2752675842,3356033026,3356139523,3557359618,3557457922,3557588994,34510798914,34510798915,34510798916,34510798920,34510807107,34510815299,34712125506,34712125507,34712125508,34712125512,34712223810,34712223811,34712223812,34712223816,34980560962,34980560963,34980659266,34980659267,34980659268,35568287746,35568287747,35568287748,35568287752,35568287756,35568287810,35568287811,35568287812,35568287816,35568287820,35568386050,35568386051,35568386052,35568386114,35568386120,35568517122,35568517123,35568517124,35568517128,35568517186,35568517187,35568517188,35568517192,35568812034,35568812035,35568812098,35568812099,35568910338,35568910339,35568910403,35569336322,35569434626,35585064962,35585064963,35585064964,35585064968,35585065026,35585065027,35585065028,35585065032,35769614338,35769614340,35769614402,35769712642,35770138626,35770236930,35786391554,35786391555,2475109646403,d9,m9,2475109711938,2475109711939,2475109720067,2475109720131,2475109728259,2475109728323,k9,F9,2887426572354,2887426572355,v9,p9,2887627898946,R9,2887627997250,C9,M9,3437182386242,3437182386243,H9,3437383712771,3437383712834,q9,E9,S9,V9,4674132967490,4674132967491,4674132975619,x9,4674334294082,4674334294083,4674334302211,4674334302275,z9,4674334392386,N9,4674602729538,4674602827842,I9,4674602958914,O9,G9,5086449827906,5086449827907,5086449827912,J9,5086651154498,5086651162627,Z9,K9,5086651252802,5086651252808,5086651260931,Y9,5086919589954,X9,5086919688258,5086919819330,5636205641794,_9,5636406968386,W9,5636407066690,5636675403842,$9,5636675502146,{l:589890,m:73760,h:4},{l:589891,m:73760,h:4},{l:598083,m:73760,h:4},{l:688194,m:73760,h:4},{l:688195,m:73760,h:4},{l:696387,m:73760,h:4},{l:819266,m:73760,h:4},{l:65602,m:73764,h:4},{l:65603,m:73764,h:4},{l:589890,m:172064,h:4},{l:589891,m:172064,h:4},{l:598083,m:172064,h:4},{l:688194,m:172064,h:4},{l:688195,m:172064,h:4},{l:696387,m:172064,h:4},{l:819266,m:172064,h:4},{l:65602,m:172068,h:4},{l:65603,m:172068,h:4},{l:589890,m:172112,h:4},{l:589891,m:172112,h:4},{l:688194,m:172112,h:4},{l:819266,m:172112,h:4},{l:65602,m:172116,h:4},{l:65608,m:172116,h:4},{l:163906,m:172116,h:4},{l:589890,m:303136,h:4},{l:688194,m:303136,h:4},{l:819266,m:303136,h:4},{l:65602,m:303140,h:4},{l:589890,m:303184,h:4},{l:688194,m:303184,h:4},{l:819266,m:303184,h:4},{l:65602,m:303188,h:4},{l:65602,m:303252,h:4},a9,tf,{l:589890,m:590112,h:4},{l:589891,m:590112,h:4},Af,ef,{l:589890,m:590160,h:4},{l:589891,m:590160,h:4},nf,{l:688194,m:590160,h:4},Pf,rf,{l:589890,m:590224,h:4},{l:589891,m:590224,h:4},{l:688131,m:590224,h:4},{l:688194,m:590224,h:4},ff,Df,{l:589890,m:590368,h:4},uf,{l:688194,m:590368,h:4},{l:696323,m:590368,h:4},of,lf,{l:819266,m:590368,h:4},Qf,Lf,{l:589890,m:590416,h:4},{l:589891,m:590416,h:4},{l:589896,m:590416,h:4},Uf,{l:688136,m:590416,h:4},{l:688194,m:590416,h:4},{l:696323,m:590416,h:4},yf,{l:819266,m:590416,h:4},{l:688130,m:590480,h:4},{l:688194,m:590480,h:4},{l:65602,m:598052,h:4},{l:65603,m:598052,h:4},{l:65602,m:598100,h:4},{l:65603,m:598100,h:4},{l:73795,m:598100,h:4},{l:163906,m:598100,h:4},{l:163907,m:598100,h:4},{l:65602,m:598164,h:4},{l:163906,m:598164,h:4},sf,hf,{l:589890,m:598304,h:4},{l:589891,m:598304,h:4},Tf,{l:688194,m:598304,h:4},bf,{l:1114178,m:598304,h:4},cf,{l:1212424,m:598304,h:4},{l:1212482,m:598304,h:4},wf,gf,{l:589890,m:688416,h:4},jf,{l:589890,m:688464,h:4},{l:589890,m:688528,h:4},Bf,df,{l:688130,m:688784,h:4},{l:65602,m:696356,h:4},{l:65602,m:696404,h:4},{l:163906,m:696404,h:4},{l:589826,m:696608,h:4},{l:1114178,m:696608,h:4},{l:1212418,m:696608,h:4},{l:589826,m:696612,h:4},{l:524355,m:590112,h:288},{l:589825,m:590112,h:288},mf,{l:589890,m:590112,h:288},{l:598019,m:590112,h:288},{l:598083,m:590112,h:288},kf,{l:589890,m:590112,h:336},{l:598019,m:590112,h:336},{l:598083,m:590112,h:336},Ff,{l:589890,m:688416,h:336},vf,{l:589890,m:688464,h:336},pf,Rf,{l:589827,m:590112,h:400},{l:589890,m:590112,h:400},{l:589891,m:590112,h:400},Cf,Mf,Hf,qf,{l:589890,m:819488,h:400},Ef,Sf,Vf,xf,{l:589827,m:819488,h:544},zf,Nf,If,{l:688199,m:688464,h:592},Of,Gf,Jf,{l:688202,m:819600,h:592},Zf,{l:589896,m:1212704,h:592},Kf,Yf,{l:688200,m:1212752,h:592},Xf,{l:688135,m:688464,h:656},{l:688135,m:819536,h:656},{l:589891,m:65936,h:8228},_f,{l:589890,m:590112,h:8480},{l:598019,m:590112,h:8480},Wf,{l:589890,m:688416,h:8480},$f,{l:589890,m:688464,h:8480},af,tD,AD,eD,{l:589896,m:1213008,h:8480},nD,{l:688200,m:1213008,h:8480},{l:688135,m:1343824,h:8480},PD,{l:589890,m:590112,h:8484},rD,{l:589890,m:590160,h:8484},{l:1212424,m:590416,h:8484},{l:1212488,m:590416,h:8484},iD,{l:688194,m:598304,h:8484},{l:589826,m:590112,h:8528},{l:589890,m:590112,h:8528}]),nt.dG=v(F(l.i7,1),w0,6,15,[.6966999769210815,0,.4885999858379364,-0.47269999980926514,-0.07490000128746033,fD,.273499995470047,.5699999928474426,.7009999752044678,.9534000158309937,-3.6435000896453857,-2.150899887084961,.4975000023841858,-2.19950008392334,-0.2809000015258789,-0.8259999752044678,-0.1784999966621399,-1.620300054550171,-1.0959999561309814,-0.3619999885559082,.13950000703334808,-0.29750001430511475,-1.2907999753952026,1.0161999464035034,DD,-2.5383999347686768,TP,.4291999936103821,-0.5824000239372253,-0.1834000051021576,.1306000053882599,-0.5015000104904175,fD,-0.5257999897003174,.4244000017642975,-0.16099999845027924,-0.2777999937534332,.5110999941825867,-0.435699999332428,-0.10409999638795853,.3424000144004822,-0.061500001698732376,.6035000085830688,.7226999998092651,.43459999561309814,-1.6821000576019287,-0.3310000002384186,-0.49799999594688416,uD,-0.3650999963283539,.45969998836517334,.3384000062942505,.6632999777793884,.4544000029563904,.15970000624656677,.633899986743927,.35040000081062317,.04490000009536743,.34200000762939453,.26109999418258667,.40459999442100525,.5218999981880188,-1.065000057220459,-1.2314000129699707,-1.802299976348877,-0.36320000886917114,-0.4108000099658966,.30570000410079956,-0.14560000598430634,-0.27129998803138733,-0.5192999839782715,.45260000228881836,.5539000034332275,.8374000191688538,-0.7070000171661377,-0.48809999227523804,-0.4099999964237213,0,.14790000021457672,.3447999954223633,.42980000376701355,.5579000115394592,-0.1264999955892563,-0.042500000447034836,.07670000195503235,.6635000109672546,-0.38119998574256897,-0.8367999792098999,1.0286999940872192,-0.10209999978542328,.3587000072002411,-0.5945000052452087,.16920000314712524,-0.121799997985363,.32829999923706055,.22390000522136688,.20430000126361847,.05900000035762787,-0.48350000381469727,.6165000200271606,-0.4011000096797943,.5577999949455261,-0.21639999747276306,-0.017500000074505806,.29809999465942383,.10999999940395355,.27149999141693115,-0.2994999885559082,-0.46700000762939453,.1565999984741211,.046799998730421066,-0.13210000097751617,1.3686000108718872,0,-0.4115999937057495,1.0185999870300293,-0.3935000002384186,.5223000049591064,.3684999942779541,.25769999623298645,1.5192999839782715,.2705000042915344,.3790999948978424,.012000000104308128,-0.33970001339912415,.14830000698566437,.2766000032424927,.35929998755455017,.7714999914169312,.3149999976158142,-1.618499994277954,.18889999389648438,-0.2651999890804291,-0.09650000184774399,.420199990272522,.18709999322891235,-0.3684000074863434,-0.07779999822378159,.8942999839782715,.3693999946117401,.28790000081062317,.4489000141620636,-0.26010000705718994,.4771000146865845,.1923000067472458,.43810001015663147,.16949999332427979,.45249998569488525,.3352000117301941,.1582999974489212,.4036000072956085,-0.04800000041723251,.5023000240325928,-0.26489999890327454,.76910001039505,-0.35519999265670776,1.0300999879837036,-0.11410000175237656,-0.5932000279426575,.17489999532699585,.13130000233650208,-0.18039999902248383,.399399995803833,.22910000383853912,.31690001487731934,.35989999771118164,-0.0038999998942017555,-0.2955999970436096,.4408999979496002,-0.16089999675750732,.3774999976158142,-0.13459999859333038,.2838999927043915,.5128999948501587,.1265999972820282,.4293999969959259,.28060001134872437,.49070000648498535,oD,.219200000166893,.15649999678134918,.6934999823570251,.3617999851703644,.6735000014305115,.5777999758720398,-0.5636000037193298,.5569000244140625,.30379998683929443,-0.32760000228881836,-0.6991999745368958,.010300000198185444,-0.4659000039100647,lD,TP,.18410000205039978,.707099974155426,.12269999831914902,.7949000000953674,-0.6592000126838684,-1.3148000240325928,-0.4066999852657318,-0.1316000074148178,-0.4925000071525574,-0.09290000051259995,-0.4352000057697296,-0.2206999957561493,-0.9959999918937683,-0.723800003528595,-0.5468999743461609,-1.2939000129699707,-0.01360000018030405,.065700002014637,.7189000248908997,.05700000002980232,.661899983882904,-0.6381000280380249,-0.8072999715805054,.23549999296665192,.30480000376701355,-0.019899999722838402,-0.07519999891519547,.44609999656677246,.1559000015258789,1.1167999505996704,-1.8039000034332275,.23649999499320984,-0.22059999406337738,.4480000138282776,-1.1339999437332153,-0.5652999877929688,-0.40529999136924744,-0.13609999418258667,.2198999971151352,.053599998354911804,-0.020999999716877937,.6984999775886536,.9642999768257141,-0.41519999504089355,-1.0369000434875488,-0.18299999833106995,.5882999897003174,-0.29179999232292175,-0.5293999910354614,-0.6541000008583069,.9473000168800354,-0.19059999287128448,-0.8483999967575073,-0.3456999957561493,.9541000127792358,1.4230999946594238,-0.7924000024795532,QD,LD,-0.2596000134944916,.8381999731063843,-0.4415999948978424,-0.37040001153945923,-0.7487000226974487,-0.10790000110864639,-0.29919999837875366,-0.3276999890804291,.025100000202655792,-0.9187999963760376,.10939999669790268,.8230999708175659,-3.233299970626831,.03500000014901161,lD,UD,.2791000008583069,.3206000030040741,.5662000179290771,-0.3783999979496002,.4032000005245209,-1.794800043106079,-0.15539999306201935,.3785000145435333,.05339999869465828,-0.16529999673366547,-0.09870000183582306,-0.10050000250339508,-0.6460999846458435,.8034999966621399,-0.24050000309944153,-0.12380000203847885,-0.35760000348091125,.09610000252723694,-0.6401000022888184,.2029000073671341,.23589999973773956,.4950999915599823,.19210000336170197,-0.37450000643730164,.34630000591278076,.289900004863739,-0.15330000221729279,-0.4169999957084656,.37700000405311584,.6998000144958496,.593999981880188,.5911999940872192,-0.5570999979972839,.023800000548362732,-0.2475000023841858,.030700000002980232,-0.38749998807907104,-0.7437000274658203,.5144000053405762,.00570000009611249,.765500009059906,.1720000058412552,-2.5624001026153564,-0.30660000443458557,.36469998955726624,.17270000278949738,-0.03290000185370445,-0.18930000066757202,.07020000368356705,-1.2453999519348145,.14959999918937683,DD,.4146000146865845,-0.2667999863624573,-0.11060000211000443,.03620000183582306,-0.3188999891281128,-0.7278000116348267,-0.08940000087022781,-0.22769999504089355,-0.2393999993801117,1.011199951171875,-0.2962000072002411,.7775999903678894,.2944999933242798,-0.22339999675750732,.27639999985694885,.8011000156402588,-0.17440000176429749,.15809999406337738,-1.7552000284194946,-0.384799987077713,.5993000268936157,.5267999768257141,-0.04170000180602074,.4733000099658966,-0.3400999903678894,-0.14499999582767487,.7088000178337097,-0.13179999589920044,.04259999841451645,-0.5027999877929688,.3831999897956848,-0.011800000444054604,-0.4357999861240387,.3749000132083893,-0.12030000239610672,-0.5648000240325928,-0.1972000002861023,-0.8769000172615051,-0.3675000071525574,-0.2003999948501587,-0.6069999933242798,-0.18569999933242798,.3467999994754791,-0.36239999532699585,.5357999801635742,-0.3700999915599823,.13359999656677246,.9545000195503235,.11389999836683273,-0.16990000009536743,TP,.2890999913215637,.2612999975681305,-0.03440000116825104,-1.9498000144958496,-2.024899959564209,-0.6004999876022339,-0.6258000135421753,-1.2348999977111816,.328000009059906,-0.54339998960495,-0.7712000012397766,-0.9057000279426575,-0.16680000722408295,-0.9904999732971191,-0.03720000013709068,-1.1638000011444092,.12620000541210175,-0.5248000025749207,-0.15379999577999115,-0.36820000410079956,.3249000012874603,.06499999761581421,.051100000739097595,-0.46070000529289246,.22310000658035278,.28220000863075256,.1396999955177307,-0.49380001425743103,.39480000734329224,-0.4074999988079071,-0.6410999894142151,-0.009100000374019146,-0.13330000638961792,-0.5192000269889832,-0.16609999537467957,TP,-0.07069999724626541,.4805999994277954,.38280001282691956,.22290000319480896,.6159999966621399,.33709999918937683,.188400000333786,.13809999823570251,uD,.2833000123500824,-0.1225999966263771,-3.9189000129699707,-0.459199994802475,-0.3434999883174896,-0.6654000282287598,-0.5055999755859375,-0.863099992275238,.15360000729560852,-0.6427000164985657,-0.08839999884366989,-0.0471000000834465,.11060000211000443,.382099986076355,-0.23919999599456787,-0.4050999879837036,.08910000324249268,-0.6972000002861023,-0.4699000120162964,.09220000356435776,.08060000091791153,-0.6773999929428101,-0.062199998646974564,-0.9300000071525574,.13369999825954437]);},nt.gG=function(t){var A,n,P,r,i;for(n=0,e.lq(t),e.bw(t,7),A=0;A<t.f;A++)try{i=e.ff(t,A,6241),r=(P=Lt.g4(nt.eG,T.Pib(i)),P<0?-1:P),r!=-1&&(n+=nt.dG[r]);}catch(D){if(D=M1(D),!Kt(D,19))throw Ut(D);}return n;},nt.hG=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g;if(r=new nt.wG(),y.sj(r.a,new nt.tG("cLogP Values are estimated applying an atom-type based increment system.",2)),y.sj(r.a,new nt.tG(yD,2)),y.sj(r.a,new nt.tG(Vr,2)),t){for(e.lq(t),e.bw(t,7),u=0,P=new y.fwb(),o=new l0.Rmb("#0.000"),A=0;A<t.f;A++)try{n=e.ff(t,A,6241),g=y.opb(P,T.Pib(n)),g?y.Zvb(P,T.Pib(n),T.Aib(g.a+1)):y.Zvb(P,T.Pib(n),T.Aib(1));}catch(B){if(B=M1(B),Kt(B,19))++u;else throw Ut(B);}for(u!=0&&y.sj(r.a,new nt.tG("Warning: "+u+" atom type(s) could not be determined.",2)),c=(D=new y.pwb(new y.vwb(new y.Cpb(P).a).b),new y.Lpb(D));y.kob(c.a.a);)b=(i=y.nwb(c.a),i.gd()),L=Lt.g4(nt.eG,b),(L<0?-1:L)!=-1?nt.vG(r,(h=b,y.fnb(y.Uvb(P,h))+" * "+l0.Omb(o,nt.dG[Q=Lt.g4(nt.eG,b),Q<0?-1:Q])+" AtomType: 0x"+T.Nib(b.a)),2):nt.vG(r,"Warning: For atom type 0x"+T.Nib(b.a)+" ("+(s=b,y.fnb(y.Uvb(P,s)))+" times found) is no increment available.",2);}return r;},nt.iG=function(){nt.fG();var t,A,n,P;if(!nt.eG&&!nt.eG)for(nt.eG=new Lt.j4(),A=nt.cG,n=0,P=A.length;n<P;++n)t=T.Pib(A[n]),Lt.d4(nt.eG,t);},N(213,1,{},nt.iG),l.s9=O(213),nt.jG=function(t,A,n,P,r){var i,D,u,o,Q,L;for(i=1/(1+d.Math.exp(t-5)),L=1-1/(1+d.Math.exp(A+5)),Q=1/(1+d.Math.exp(.012*n-6)),D=1-1/(1+d.Math.exp(P)),u=(.5+i/2)*(.5+L/2)*(.5+Q/2)*(.5+D/2),o=0;r!=null&&o<r.length;o++)r[o]==2?u*=.8:r[o]==3&&(u*=.6);return u;},nt.mG=function(t,A){var n,P,r,i,D,u,o,Q,L;if(n=new nt.wG(),!nt.lG)return y.sj(n.a,new nt.tG(sD,2)),-999;for(y.sj(n.a,new nt.tG("Found sub-structure fragments and their contributions:",2)),y.sj(n.a,new nt.tG("(yellow atoms carry at least one more substituent)",2)),Q=0,o=0,i=0,L=new e.ju(1),r=new e.tw(),D=0;D<nt.kG.a.a.length;D++)e.Tq(new e._q(false),r,nt.pG(nt.kG,D)),e.cu(L,A),e._t(L,r),e.Pt(L,1,L.c)>0&&(u=nt.qG(nt.kG,D),u<-1?Q+=u:(o+=u,++i),nt.vG(n,nt.pG(nt.kG,D),1),y.sj(n.a,new nt.tG(""+u,3)));return i==0?-1:(P=Q+o/d.Math.sqrt(i),P=P+.0625*(i-40),t.a=n,P);},nt.nG=function(t){var A,n,P,r,i,D;if(!nt.lG)return sD;for(i=0,P=0,D=new e.ju(1),n=new e.tw(),r=0;r<nt.kG.a.a.length;r++)e.Tq(new e._q(false),n,nt.pG(nt.kG,r)),e.cu(D,t),e._t(D,n),e.Pt(D,1,D.c)>0&&(i+=nt.qG(nt.kG,r),++P);return A=P==0?-1:i/d.Math.sqrt(P),A+"	"+P+"	"+t.f;},nt.oG=function(){if(!nt.lG)try{nt.kG=new nt.rG(),nt.lG=!0;}catch(t){if(t=M1(t),Kt(t,19))T.Zjb();else throw Ut(t);}},N(273,1,{},nt.oG),nt.lG=false,l.t9=O(273),nt.pG=function(t,A){return y.xj(t.a,A).a;},nt.qG=function(t,A){return y.xj(t.a,A).b;},nt.rG=function(){var t,A,n,P,r,i;for(i=new Ct.ogb(new Ct.tgb((n=_t.mAb("/resources/druglikenessNoIndex.txt"),S1.Mmb(),n))),this.a=new y.Mj();r=Ct.ngb(i),r!=null;){if(P=T.rjb(r,Y1(9)),P==-1)throw Ut(new T.cC("line without TAB"));t=(w.Wzb(0,P,w.aAb(r).length),w.aAb(r).substr(0,P)),A=T._hb(w.aAb(T.Vhb((w.Xzb(P+1,w.aAb(r).length+1),w.aAb(r).substr(P+1))))),y.sj(this.a,new nt.sG(t,A));}},N(274,1,{},nt.rG),l.v9=O(274),nt.sG=function(t,A){this.a=t,this.b=A;},N(236,1,{236:1},nt.sG),U.b=0,l.u9=O(236),nt.tG=function(t,A){this.a=t,this.b=A;},N(45,1,{45:1},nt.tG),U.b=0,l.x9=O(45),nt.uG=function(t){t.a=new y.Mj();},nt.vG=function(t,A,n){y.sj(t.a,new nt.tG(A,n));},nt.wG=function(){nt.uG(this);},N(132,1,{},nt.wG),l.w9=O(132),nt.zG=function(){nt.zG=D1,nt.xG=v(F(l.Ybb,1),st,2,6,["[N](-*)(-*)-*","[N](-*)=*","[N]#*","[N](-*)(=*)=* as in nitro","[N](=*)#* middle atom of azide","[N]1(-*)-*-*-1 3-membered ring","[NH](-*)-*","[NH]1-*-*-1 3-membered ring","[NH]=*","[NH2]-*","[N+](-*)(-*)(-*)-*","[N+](-*)(-*)=*","[N+](-*)#* N in isocyano","[NH+](-*)(-*)-*","[NH+](-*)=*","[NH2+](-*)-*","[NH2+]=*","[NH3+]-*","[n](:*):*","[n](:*)(:*):*","[n](-*)(:*):*","[n](=*)(:*):* as in pyridine-N-oxid","[nH](:*):*","[n+](:*)(:*):*","[n+](-*)(:*):*","[nH+](:*):*","[O](-*)-*","[O]1-*-*-1 3-membered ring","[O]=*","[OH]-*","[O-]-*","[o](:*):*","[S](-*)-*","[S]=*","[S](-*)(-*)=*","[S](-*)(-*)(=*)=*","[SH]-*","[s](:*):*","[s](=*)(:*):*","[P](-*)(-*)-*","[P](-*)=*","[P](-*)(-*)(-*)=*","[PH](-*)(-*)=*"]),nt.yG=v(F(l.i7,1),w0,6,15,[3.240000009536743,12.359999656677246,23.790000915527344,11.680000305175781,13.600000381469727,hD,12.029999732971191,21.940000534057617,23.850000381469727,26.020000457763672,0,hD,4.360000133514404,4.440000057220459,13.970000267028809,16.610000610351562,25.59000015258789,27.639999389648438,12.890000343322754,4.409999847412109,4.929999828338623,8.390000343322754,15.789999961853027,4.099999904632568,3.880000114440918,14.140000343322754,9.229999542236328,12.529999732971191,17.06999969482422,20.229999542236328,23.059999465942383,13.140000343322754,25.299999237060547,32.09000015258789,19.209999084472656,8.380000114440918,38.79999923706055,28.239999771118164,21.700000762939453,13.59000015258789,34.13999938964844,9.8100004196167,23.469999313354492]);},nt.AG=function(t,A){var n,P;for(n=nt.EG(t),P=0;P<nt.yG.length;P++)n[P]!=0&&nt.vG(A,""+n[P]+" * "+nt.yG[P]+"   AtomType: "+nt.xG[P],2);},nt.BG=function(t){var A,n,P;for(A=nt.EG(t),P=0,n=0;n<nt.yG.length;n++)P+=A[n]*nt.yG[n];return P;},nt.CG=function(t){var A;return A=new nt.wG(),y.sj(A.a,new nt.tG("The polar surface area prediction is based on an atom-type based",2)),y.sj(A.a,new nt.tG("increment system, published by P. Ertl, B. Rohde, P. Selzer",2)),y.sj(A.a,new nt.tG("in J. Med. Chem. 2000, 43, 3714-3717",2)),y.sj(A.a,new nt.tG(Vr,2)),nt.AG(t,A),A;},nt.DG=function(t,A){var n;switch(t.C[A]){case 7:if(A<t.f&&e.ot(t.p,A)){if(t.s[A]==0){if(t.e[A]-e.tp(t,A)+e.op(t,A)==0){if(e.tp(t,A)==2)return 18;for(n=0;n<e.tp(t,A);n++)if(!e.Up(t,t.k[A][n]))return 20;return 19;}else return 22;}else if(t.s[A]==1)if(t.e[A]-e.tp(t,A)+e.op(t,A)==0){for(n=0;n<e.tp(t,A);n++)if(!e.Up(t,t.k[A][n]))return e.Bl(t,t.i[A][n])<0?21:24;return 23;}else return 25;}else if(t.s[A]==0)switch(t.e[A]-e.tp(t,A)+e.op(t,A)){case 0:switch(t.o[A]){case 0:return (t.p&&A<t.f?e.dt(t.p,A):0)==3?5:0;case 1:return 1;case 2:return 2;}break;case 1:switch(t.o[A]){case 0:return (t.p&&A<t.f?e.dt(t.p,A):0)==3?7:6;case 1:return 8;}break;case 2:return 9;}else if(t.s[A]==1)switch(t.e[A]-e.tp(t,A)+e.op(t,A)){case 0:switch(t.o[A]){case 0:return 10;case 1:return nt.FG(t,A)?3:11;case 2:return t.n[A][0]==2?nt.FG(t,A)?4:nt.yG.length+1:12;}break;case 1:switch(t.o[A]){case 0:return 13;case 1:return 14;}break;case 2:return t.o[A]==0?15:16;case 3:return 17;}return nt.yG.length+1;case 8:if(A<t.f&&e.ot(t.p,A)){if(t.s[A]==0)return 31;}else {if(t.s[A]==0)return t.o[A]>0?28:e.tp(t,A)==1?29:(t.p&&A<t.f?e.dt(t.p,A):0)==3?27:26;if(t.s[A]==-1)return e.tp(t,A)==1&&e.Bl(t,t.i[A][0])>0?28:30;}return nt.yG.length+1;case 15:if(t.s[A]==0){if(t.e[A]-e.tp(t,A)+e.op(t,A)==0){if(e.tp(t,A)==3&&t.o[A]==0)return 39;if(e.tp(t,A)==2&&t.o[A]==1)return 40;if(e.tp(t,A)==4&&t.o[A]==1)return 41;}else if(t.e[A]-e.tp(t,A)+e.op(t,A)==1&&e.tp(t,A)==3&&t.o[A]==1)return 42;}return nt.yG.length+1;case 16:if(t.s[A]==0){if(A<t.f&&e.ot(t.p,A))return e.tp(t,A)==2?37:38;if(t.e[A]-e.tp(t,A)+e.op(t,A)==0){if(e.tp(t,A)==2&&t.o[A]==0)return 32;if(e.tp(t,A)==1&&t.o[A]==1)return 33;if(e.tp(t,A)==3&&t.o[A]==1)return 34;if(e.tp(t,A)==4&&t.o[A]==2)return 35;}else if(t.e[A]-e.tp(t,A)+e.op(t,A)==1&&e.tp(t,A)==1)return 36;}return nt.yG.length+1;}return nt.yG.length;},nt.EG=function(t){var A,n;for(n=j(l.j7,H,6,nt.yG.length+2,15,1),e.bw(t,7),A=0;A<t.f;A++)++n[nt.DG(t,A)];return n;},nt.FG=function(t,A){var n;for(n=0;n<t.j[A];n++)if(e.Bl(t,t.i[A][n])<0)return  true;return  false;},nt.GG=function(t){var A,n;for(n=0,A=0;A<t.a.q;A++)(e.Tl(t.a,A)==7||e.Tl(t.a,A)==8)&&++n;return n;},nt.HG=function(t){var A,n;for(n=0,A=0;A<t.a.q;A++)(e.Tl(t.a,A)==7||e.Tl(t.a,A)==8)&&e.No(t.a,A)>0&&++n;return n;},nt.IG=function(t){var A;try{return nt.gG((new nt.iG(),t.a));}catch(n){if(n=M1(n),Kt(n,19))return A=n,T.MB(A,(T.Zjb(),T.Yjb),""),-999;throw Ut(n);}},nt.JG=function(t){this.a=t,e.lq(t);},N(278,1,{}),l.y9=O(278),nt.MG=function(){nt.MG=D1,nt.KG=v(F(l.k7,1),e0,6,14,[524290,524292,524297,524301,589826,589827,589828,589829,589832,589833,589837,589838,688130,688131,688132,688137,688141,819202,819203,819204,819205,819208,819209,1114114,1114115,1114116,1114120,1212420,1343492,1343496,1638402,1638403,17367042,17367043,17367044,17367045,17367048,17367049,17367053,17367054,17465346,1074266116,1208483842,1208483844,1208549378,1208549379,1208549380,1208549384,1208549388,1409810434,1409875970,1409875971,1409875972,1409875976,1409974274,1409974280,1409974284,1678311426,1678311427,1678311428,1678311432,1678311436,1678409730,1678409731,1678409732,1678540802,1678540803,1678540804,1678540808,2282291202,2282291203,2282389506,2282389507,2282520578,2282520579,2483617794,2483617795,2483716098,2483716099,2752675842,3356033026,3557359618,3557457922,3557588994,34510798850,34510798851,34510798852,34510798856,34712125442,34712125443,34712125444,34712125448,34712223746,34712223747,34712223748,34980560898,34980560899,34980659202,35568287746,35568287747,35568287748,35568287752,35568386050,35568386051,35568386052,35568517122,35568517123,35568517124,35568517128,35568812034,35568812035,35568910338,35568910339,35568910350,35569336322,35585064962,35585064963,35585064964,35585064968,35769548802,35769614338,35769712642,35769843714,35786391554,d9,m9,k9,F9,v9,p9,R9,C9,M9,H9,q9,E9,S9,3437652377603,V9,x9,4674334294019,z9,N9,4674602827778,I9,O9,G9,J9,5086651154435,Z9,5086651252739,K9,5086651252744,Y9,5086919589896,X9,5086919819266,_9,W9,$9,5636675633154,{l:589826,m:73760,h:4},{l:589827,m:73760,h:4},{l:688130,m:73760,h:4},{l:819202,m:73760,h:4},{l:65538,m:73764,h:4},{l:65539,m:73764,h:4},{l:589826,m:172064,h:4},{l:589827,m:172064,h:4},{l:688130,m:172064,h:4},{l:819202,m:172064,h:4},{l:819203,m:172064,h:4},{l:65538,m:172068,h:4},{l:65539,m:172068,h:4},{l:589826,m:172112,h:4},{l:589827,m:172112,h:4},{l:688130,m:172112,h:4},{l:819202,m:172112,h:4},{l:65538,m:172116,h:4},{l:163842,m:172116,h:4},{l:589826,m:303136,h:4},{l:688130,m:303136,h:4},{l:819202,m:303136,h:4},{l:65538,m:303140,h:4},{l:589826,m:303184,h:4},{l:688130,m:303184,h:4},{l:819202,m:303184,h:4},{l:65538,m:303188,h:4},a9,tf,Af,ef,nf,Pf,rf,{l:688130,m:590224,h:4},{l:819202,m:590224,h:4},ff,Df,uf,of,lf,Qf,Lf,Uf,{l:688135,m:590416,h:4},yf,{l:1212430,m:590416,h:4},{l:65538,m:598052,h:4},{l:65538,m:598100,h:4},{l:65539,m:598100,h:4},{l:163842,m:598100,h:4},{l:163843,m:598100,h:4},sf,hf,Tf,{l:688131,m:598304,h:4},bf,cf,wf,gf,jf,{l:589826,m:688672,h:4},Bf,df,mf,kf,{l:589827,m:590112,h:336},Ff,vf,pf,Rf,Cf,Mf,Hf,qf,Ef,{l:688130,m:819536,h:400},Sf,Vf,xf,{l:589831,m:590112,h:592},zf,Nf,If,Of,Gf,Jf,{l:819207,m:819600,h:592},Zf,Kf,Yf,{l:589832,m:1212816,h:592},{l:688136,m:1212816,h:592},Xf,_f,Wf,$f,af,{l:589826,m:819488,h:8480},tD,AD,eD,nD,{l:819208,m:1213008,h:8480},PD,rD,{l:688130,m:590160,h:8484},{l:589826,m:598304,h:8484},iD,{l:819202,m:598304,h:8484}]),nt.LG=v(F(l.i7,1),w0,6,15,[-0.1899999976158142,1.2699999809265137,-0.7009999752044678,2.690999984741211,-0.22699999809265137,Di,.10599999874830246,-0.47600001096725464,-0.44699999690055847,-0.19099999964237213,-0.3330000042915344,.0860000029206276,.24699999392032623,-0.06199999898672104,.01600000075995922,.3869999945163727,.23499999940395355,-0.4320000112056732,-0.902999997138977,.38999998569488525,.5809999704360962,4.52400016784668,-0.6349999904632568,-0.2800000011920929,.7699999809265137,-0.05000000074505806,1.0870000123977661,.19200000166893005,.19599999487400055,-0.5199999809265137,.5419999957084656,.3630000054836273,.7919999957084656,.5920000076293945,.9639999866485596,oD,-0.6850000023841858,-0.3149999976158142,-0.4129999876022339,-0.5950000286102295,.2199999988079071,-1.4320000410079956,-2.253999948501587,.4399999976158142,-0.27000001072883606,-0.13300000131130219,-0.26899999380111694,.2669999897480011,.5720000267028809,-0.5680000185966492,.17399999499320984,-0.1850000023841858,-0.23499999940395355,TD,bD,-0.34200000762939453,-0.3479999899864197,-0.43700000643730164,-0.8040000200271606,-0.41200000047683716,-0.2150000035762787,-0.625,-0.8309999704360962,.4970000088214874,-0.4309999942779541,-1.3309999704360962,.5070000290870667,-0.6320000290870667,-0.5989999771118164,-0.15600000321865082,-0.3529999852180481,-0.164000004529953,-0.4410000145435333,-0.4970000088214874,-1.059999942779541,.11500000208616257,-0.22499999403953552,-0.15399999916553497,-0.03099999949336052,-1.5740000009536743,-1.093000054359436,UD,-0.44999998807907104,-0.5559999942779541,bD,2.384000062942505,1.75,-1.6660000085830688,-1.065999984741211,1.3270000219345093,.8029999732971191,-1.5049999952316284,-2.5369999408721924,TD,.14900000393390656,.5210000276565552,2.9049999713897705,-0.25200000405311584,.8600000143051147,.3610000014305115,.40299999713897705,.004999999888241291,1.1460000276565552,.9359999895095825,-0.30000001192092896,.20900000631809235,-0.5830000042915344,-0.024000000208616257,-0.009999999776482582,.6110000014305115,.4860000014305115,.8619999885559082,-0.03500000014901161,-0.5960000157356262,1.1610000133514404,1.6469999551773071,.843999981880188,.125,.1420000046491623,-0.17100000381469727,.44200000166893005,.08799999952316284,3.065999984741211,1.6519999504089355,cD,-0.017999999225139618,.023000000044703484,.0729999989271164,.2540000081062317,.5540000200271606,.5950000286102295,-0.4059999883174896,-0.6370000243186951,-0.17399999499320984,-0.10100000351667404,-0.5429999828338623,-2.4059998989105225,-3.2920000553131104,-0.05299999937415123,-0.19300000369548798,Hr,-1.2610000371932983,-0.656000018119812,-0.7300000190734863,-0.9380000233650208,.12800000607967377,1.1540000438690186,.24199999868869781,-0.5289999842643738,-0.27799999713897705,-0.8019999861717224,.9120000004768372,-1.38100004196167,.46299999952316284,1.0740000009536743,-0.628000020980835,-0.9620000123977661,-1.8320000171661377,-1.4989999532699585,-2.115999937057495,-2.2070000171661377,-1.3170000314712524,2.500999927520752,-0.8489999771118164,QD,-0.621999979019165,2.121999979019165,-2.2260000705718994,.9129999876022339,-2.259000062942505,-1.25,1.3940000534057617,-1.4019999504089355,2.072999954223633,-2.9570000171661377,.29100000858306885,-3.4760000705718994,-2.7269999980926514,-3.131999969482422,-2.119999885559082,-0.7250000238418579,-0.296999990940094,.08299999684095383,.34700000286102295,-1.4249999523162842,-1.659999966621399,-1.281999945640564,-1.2649999856948853,.718999981880188,.1379999965429306,1.3020000457763672,.859000027179718,1.3589999675750732,.6589999794960022,-0.9399999976158142,.8999999761581421,.3190000057220459,-2.571000099182129,1.1089999675750732,.972000002861023,1.652999997138977,2.6019999980926514,.7289999723434448,1.065999984741211,1.0670000314712524,-0.3109999895095825,.03099999949336052,cD,-0.6809999942779541,-1.2580000162124634,Ei,-3.0959999561309814,-0.2280000001192093,1.9329999685287476,.11900000274181366,2.1080000400543213,.11299999803304672,1.628000020980835,1.3079999685287476,3.3359999656677246,.7540000081062317,-0.4650000035762787,-0.3970000147819519,.07699999958276749,-0.4790000021457672,-0.15299999713897705,.14100000262260437,2.134999990463257,.23399999737739563,.460999995470047,.6700000166893005,-0.3610000014305115,-1.0390000343322754,-0.4830000102519989,.13699999451637268,-0.7680000066757202,-0.5109999775886536,3.4240000247955322,-0.8550000190734863,-0.5849999785423279,-1.5670000314712524,3.3429999351501465,1.840000033378601,.3889999985694885,1.121999979019165,qr,1.3350000381469727,.3659999966621399,-0.5569999814033508,.4320000112056732,.20399999618530273,.8820000290870667,.4659999907016754,-0.4580000042915344,.40400001406669617,.6570000052452087,1.1150000095367432,1.9759999513626099,1.7860000133514404,-0.035999998450279236,-1.0499999523162842,1.0449999570846558,.04399999976158142,1.0329999923706055,-1.0800000429153442,2.5390000343322754,2.234999895095825,vr,3.121000051498413,3.931999921798706,2.75]);},nt.NG=function(t){var A,n,P,r;for(P=-0.5299999713897705,e.lq(t),e.bw(t,7),A=0;A<t.f;A++){r=-1;try{r=e.ff(t,A,2144);}catch(i){if(i=M1(i),!Kt(i,19))throw Ut(i);}for(n=0;n<nt.LG.length;n++)if(H1(nt.KG[n],r)){P+=nt.LG[n];break;}}return P;},nt.OG=function(t){var A,n,P,r,i,D,u;if(P=new nt.wG(),y.sj(P.a,new nt.tG("Solubility values are estimated applying an atom-type based increment system.",2)),y.sj(P.a,new nt.tG(yD,2)),y.sj(P.a,new nt.tG(Vr,2)),y.sj(P.a,new nt.tG("Base value = -0.530",2)),n=j(l.j7,H,6,nt.LG.length,15,1),t)for(e.lq(t),e.bw(t,7),A=0;A<t.f;A++){u=-1;try{u=e.ff(t,A,2144);}catch(o){if(o=M1(o),!Kt(o,19))throw Ut(o);}for(D=0;D<nt.LG.length;D++)if(H1(nt.KG[D],u)){++n[D];break;}}for(r=new l0.Rmb("#0.000"),i=0;i<nt.LG.length;i++)n[i]!=0&&nt.vG(P,""+n[i]+" * "+l0.Omb(r,nt.LG[i])+"   AtomType: 0x"+T.Nib(nt.KG[i]),2);return P;},nt.VG=function(){nt.VG=D1,nt.PG=v(F(l.Ybb,1),st,2,6,["mutagenic","tumorigenic","irritant","reproductive effective"]),nt.QG=v(F(l.Ybb,1),st,2,6,["Mutagenicity","Tumorigenicity","Irritating effects","Reproductive effects"]);},nt.WG=function(t,A){var n,P,r,i;if(!nt.SG)return 0;if(nt.UG[A].bc(e.rg(new e.Ng(t)))!=-1)return 3;for(i=new e.ju(1),n=new e.tw(),r=0;r<nt.RG[A].a.length;r++)if(e.Tq(new e._q(false),n,y.xj(nt.RG[A],r)),e.cu(i,t),e._t(i,n),e.Pt(i,1,i.c)>0)return 3;for(P=0;P<nt.TG[A].a.length;P++)if(e.Tq(new e._q(false),n,y.xj(nt.TG[A],P)),e.cu(i,t),e._t(i,n),e.Pt(i,1,i.c)>0)return 2;return 1;},nt.XG=function(t,A){var n,P,r,i,D,u,o;if(o=new nt.wG(),!nt.SG)return y.sj(o.a,new nt.tG("Toxicity predictor not properly initialized.",2)),o;if(D=e.rg(new e.Ng(t)),nt.UG[A].bc(D)!=-1)return nt.vG(o,"This molecule is known to be "+nt.PG[A]+":",2),y.sj(o.a,new nt.tG(D,1)),o;for(u=new e.ju(1),n=false,P=new e.tw(),i=0;i<nt.RG[A].a.length;i++)e.Tq(new e._q(false),P,y.xj(nt.RG[A],i)),e.cu(u,t),e._t(u,P),e.Pt(u,1,u.c)>0&&(n||nt.vG(o,"High-risk fragments indicating "+nt.QG[A]+":",2),n=true,nt.vG(o,y.xj(nt.RG[A],i),1));for(n=false,r=0;r<nt.TG[A].a.length;r++)e.Tq(new e._q(false),P,y.xj(nt.TG[A],r)),e.cu(u,t),e._t(u,P),e.Pt(u,1,u.c)>0&&(n||nt.vG(o,"Medium-risk fragments indicating "+nt.QG[A]+":",2),n=true,nt.vG(o,y.xj(nt.TG[A],r),1));return o.a.a.length==0&&nt.vG(o,"No indication for "+nt.QG[A]+" found.",2),o;},nt.YG=function(t){var A,n,P,r;for(r=new Ct.ogb(new Ct.tgb((n=_t.mAb(t),S1.Mmb(),n))),P=new e.Zv();;)try{if(A=Ct.ngb(r),A!=null)e.Xv(P,A);else break;}catch(i){if(i=M1(i),Kt(i,75))break;throw Ut(i);}return P;},nt.ZG=function(t){var A,n,P,r;for(r=new Ct.ogb(new Ct.tgb((P=_t.mAb(t),S1.Mmb(),P))),A=new y.Mj();;)try{if(n=Ct.ngb(r),n!=null)w.szb(A.a,n);else break;}catch(i){if(i=M1(i),Kt(i,75))break;throw Ut(i);}return A;},nt.$G=function(){if(nt.VG(),!nt.SG)try{nt.UG=j(l.i8,i1,102,4,0,1),nt.RG=j(l.Fcb,i1,20,4,0,1),nt.TG=j(l.Fcb,i1,20,4,0,1),nt.RG[0]=nt.ZG("/resources/toxpredictor/m1.txt"),nt.RG[1]=nt.ZG("/resources/toxpredictor/t1.txt"),nt.RG[2]=nt.ZG("/resources/toxpredictor/i1.txt"),nt.RG[3]=nt.ZG("/resources/toxpredictor/r1.txt"),nt.TG[0]=nt.ZG("/resources/toxpredictor/m2.txt"),nt.TG[1]=nt.ZG("/resources/toxpredictor/t2.txt"),nt.TG[2]=nt.ZG("/resources/toxpredictor/i2.txt"),nt.TG[3]=nt.ZG("/resources/toxpredictor/r2.txt"),nt.UG[0]=nt.YG("/resources/toxpredictor/m3.txt"),nt.UG[1]=nt.YG("/resources/toxpredictor/t3.txt"),nt.UG[2]=nt.YG("/resources/toxpredictor/i3.txt"),nt.UG[3]=nt.YG("/resources/toxpredictor/r3.txt"),nt.SG=!0;}catch(t){if(t=M1(t),Kt(t,19))T.Zjb();else throw Ut(t);}},N(284,1,{},nt.$G),nt.SG=false,l.z9=O(284),At._G=function(t,A){y.sj(t.a,A);},At.aH=function(t,A,n){y.rj(t.a,n,A);},At.bH=function(t,A){y.sj(t.f,A),t.d=-1;},At.cH=function(t,A,n){y.rj(t.f,n,A),t.d=-1;},At.dH=function(t,A){y.sj(t.g,A),t.d=-1;},At.eH=function(t,A,n){y.rj(t.g,n,A),t.d=-1;},At.fH=function(t){w.uzb(t.g.a,0),w.uzb(t.f.a,0),w.uzb(t.a.a,0),t.b=null,t.d=-1;},At.gH=function(t){var A,n,P;for(P=new y.Bqb(t.g);P.a<P.c.a.length;)if(A=y.Aqb(P),A.K)return  true;for(n=new y.Bqb(t.f);n.a<n.c.a.length;)if(A=y.Aqb(n),A.K)return  true;return  false;},At.hH=function(t){var A,n,P,r;for(n=0,A=0,P=0;P<t.g.a.length+t.f.a.length;P++)r=P<t.g.a.length?y.xj(t.g,P):y.xj(t.f,P-t.g.a.length),r.r!=0&&(n+=r.r,A+=e.Wl(r,r.q,r.r,(e.Rk(),e.Qk))*r.r);return n==0?(e.Rk(),e.Rk(),e.Qk):A/n;},At.iH=function(t,A){return y.xj(t.a,A);},At.jH=function(t){var A,n,P;if(t.d!=-1)return t.d;for(t.d=0,n=0;n<t.g.a.length+t.f.a.length;n++)for(P=n<t.g.a.length?y.xj(t.g,n):y.xj(t.f,n-t.g.a.length),A=0;A<P.q;A++)t.d<d.Math.abs(P.w[A])&&(t.d=d.Math.abs(P.w[A]));return t.d;},At.kH=function(t){var A,n,P,r;if(n=new At.zH(),t.g.a.length!=0){for(r=new e.vw(y.xj(t.g,0)),A=1;A<t.g.a.length;A++)e.Yk(r,y.xj(t.g,A));y.sj(n.g,r),n.d=-1;}if(t.f.a.length!=0){for(P=new e.vw(y.xj(t.f,0)),A=1;A<t.f.a.length;A++)e.Yk(P,y.xj(t.f,A));y.sj(n.f,P),n.d=-1;}return n;},At.lH=function(t,A){return A<t.g.a.length?y.xj(t.g,A):y.xj(t.f,A-t.g.a.length);},At.mH=function(t){return t.g.a.length+t.f.a.length;},At.nH=function(t){return t.e==null?"":t.e;},At.oH=function(t,A){return y.xj(t.f,A);},At.pH=function(t,A){return y.xj(t.g,A);},At.qH=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p;if(t.g.a.length==0||t.f.a.length==0||(h=At.jH(t),h==0))return null;for(s=j(l.j7,E1,7,t.f.a.length,0,2),i=0;i<t.f.a.length;i++)for(g=y.xj(t.f,i),e.bw(g,15),s[i]=j(l.j7,H,6,h+1,15,1),y.Nqb(s[i],-1),A=0;A<g.q;A++){if(L=d.Math.abs(g.w[A]),L!=0&&s[i][L]!=-1)return null;s[i][L]=A;}for(D=j(l.Feb,ht,6,h+1,16,1),r=0;r<t.g.a.length;r++)for(p=y.xj(t.g,r),e.bw(p,15),k=0;k<p.q;k++)if(L=d.Math.abs(p.w[k]),L!=0&&!D[L]){for(u=0;u<t.f.a.length;u++)if(b=s[u][L],b!=-1){if(g=y.xj(t.f,u),p.j[k]!=g.j[b]){D[L]=true;break;}if((p.u[k]&3)!=(g.u[b]&3)){D[L]=true;break;}for(o=0;o<p.j[k];o++)if(n=e.Ll(p,p.i[k][o]),n==0)D[L]=true;else {for(M=p.k[k][o],P=false,Q=0;Q<g.j[b];Q++){if(B=e.Ll(g,g.i[b][Q]),B==0){D[L]=true;break;}if(B==n){if(P=true,c=g.k[b][Q],(M<p.g&&(e.ut(p.p,M)||p.H[M]==8))^(c<g.g&&(e.ut(g.p,c)||g.H[c]==8))||!(M<p.g&&(e.ut(p.p,M)||p.H[M]==8))&&(e.fm(p,M)!=e.fm(g,c)||(p.F[M]&3)!=(g.F[c]&3))){D[L]=true,D[n]=true;break;}break;}}P||(D[L]=true);}}}return D;},At.rH=function(t){var A,n,P,r;for(P=new y.Bqb(t.g);P.a<P.c.a.length;)if(A=y.Aqb(P),A.q!=0)return  false;for(r=new y.Bqb(t.f);r.a<r.c.a.length;)if(A=y.Aqb(r),A.q!=0)return  false;for(n=new y.Bqb(t.a);n.a<n.c.a.length;)if(A=y.Aqb(n),A.q!=0)return  false;return  true;},At.sH=function(t){return t.c||At.gH(t);},At.tH=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(n=0,s=new y.Bqb(t.g);s.a<s.c.a.length;)if(Q=y.Aqb(s),e.bw(Q,1),Q.K){for(A=0;A<Q.f;A++)if(pt(J(Q.B[A],r0),0)){if(d.Math.abs(Q.w[A])!=0)return  false;}else ++n;}else n+=Q.f;for(o=new y.Bqb(t.f);o.a<o.c.a.length;)if(D=y.Aqb(o),e.bw(D,1),D.K){for(A=0;A<D.f;A++)if(pt(J(D.B[A],r0),0)){if(d.Math.abs(D.w[A])!=0)return  false;}else --n;}else n-=D.f;if(n!=0)return  false;for(i=At.jH(t),P=j(l.Feb,ht,6,i+1,16,1),L=new y.Bqb(t.g);L.a<L.c.a.length;)for(Q=y.Aqb(L),A=0;A<Q.f;A++)if(!Q.K||H1(J(Q.B[A],r0),0)){if(r=d.Math.abs(Q.w[A]),P[r])return  false;P[r]=true;}for(u=new y.Bqb(t.f);u.a<u.c.a.length;)for(D=y.Aqb(u),e.bw(D,1),A=0;A<D.f;A++)if(!D.K||H1(J(D.B[A],r0),0)){if(r=d.Math.abs(D.w[A]),r>=i||!P[r])return  false;P[r]=false;}return  true;},At.uH=function(t){var A,n,P,r;if(t.g.a.length+t.f.a.length<=1)return  false;for(A=At.hH(t),r=j(l.hab,C1,35,t.g.a.length+t.f.a.length,0,1),n=0;n<t.g.a.length+t.f.a.length;n++)if(r[n]=e.km(n<t.g.a.length?y.xj(t.g,n):y.xj(t.f,n-t.g.a.length),null),r[n]){for(P=0;P<n;P++)if(r[P]&&(r[n].c+r[n].b>=r[P].c&&r[n].c<=r[P].c+r[P].b||r[n].d+r[n].a>=r[P].d&&r[n].d<=r[P].d+r[P].a))return  true;if(n!=0&&r[n-1]&&(r[n].c-r[n-1].c-r[n].b>5*A||r[n].d-r[n-1].d-r[n].a>5*A))return  true;}return  false;},At.vH=function(t,A){t.b=A;},At.wH=function(t,A){var n,P,r;for(t.c=A,r=new y.Bqb(t.g);r.a<r.c.a.length;)n=y.Aqb(r),e.On(n,A);for(P=new y.Bqb(t.f);P.a<P.c.a.length;)n=y.Aqb(P),e.On(n,A);},At.xH=function(t,A){t.e=A;},At.yH=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c;for(u=At.jH(t),D=j(l.Feb,ht,6,u+1,16,1),c=new y.Bqb(t.g);c.a<c.c.a.length;)for(h=y.Aqb(c),n=0;n<h.q;n++)if(r=d.Math.abs(h.w[n]),r!=0){if(D[r])throw Ut(new T.cC("Duplicate mapping no in reactants: "+r));D[r]=true;}for(i=j(l.Feb,ht,6,u+1,16,1),s=new y.Bqb(t.f);s.a<s.c.a.length;)for(Q=y.Aqb(s),n=0;n<Q.q;n++)if(r=d.Math.abs(Q.w[n]),r!=0){if(i[r])throw Ut(new T.cC("Duplicate mapping no in products: "+r));i[r]=true;}for(o=j(l.j7,H,6,u+1,15,1),P=0,A=1;A<=u;A++)D[A]&&i[A]&&(o[A]=++P);if(P!=u){for(b=new y.Bqb(t.g);b.a<b.c.a.length;)for(h=y.Aqb(b),n=0;n<h.q;n++)e.rn(h,n,o[d.Math.abs(h.w[n])],h.w[n]<0);for(L=new y.Bqb(t.f);L.a<L.c.a.length;)for(Q=y.Aqb(L),n=0;n<Q.q;n++)e.rn(Q,n,o[d.Math.abs(Q.w[n])],Q.w[n]<0);}},At.zH=function(){this.g=new y.Mj(),this.f=new y.Mj(),this.a=new y.Mj(),this.d=-1,this.c=false;},At.AH=function(t){var A,n,P,r,i,D;for(At.zH.call(this),D=t&&t.g?t.g.a.length:0,i=t&&t.f?t.f.a.length:0,A=t&&t.a?t.a.a.length:0,P=0;P<D;P++)y.sj(this.g,new e.vw(y.xj(t.g,P)));for(r=0;r<i;r++)y.sj(this.f,new e.vw(y.xj(t.f,r)));for(n=0;n<A;n++)y.sj(this.a,new e.vw(y.xj(t.a,n)));this.b=new e.gk(t.b),t.e!=null&&(this.e=t.e),this.c=t.c||At.gH(t);},At.BH=function(t,A){At.zH.call(this),t!=null&&(y.uj(this.g,new y.vob(new y.nrb(t),0,A)),y.uj(this.f,new y.vob(new y.nrb(t),A,t.length))),this.c=At.gH(this);},N(53,1,ie,At.zH,At.AH,At.BH),U.c=false,U.d=0,l.B9=O(53),At.CH=function(t,A,n){var P,r,i;return i=d.Math.sqrt((t.i[0].a-A)*(t.i[0].a-A)+(t.i[0].b-n)*(t.i[0].b-n)),i<8?1:(r=d.Math.sqrt((t.i[1].a-A)*(t.i[1].a-A)+(t.i[1].b-n)*(t.i[1].b-n)),r<8?2:(P=d.Math.sqrt((t.i[1].a-t.i[0].a)*(t.i[1].a-t.i[0].a)+(t.i[1].b-t.i[0].b)*(t.i[1].b-t.i[0].b)),i+r<P+5?3:0));},At.DH=function(t){var A,n;return A=Rt(t.i[1].a-t.i[0].a),n=Rt(t.i[1].b-t.i[0].b),d.Math.sqrt(A*A+n*n);},At.EH=function(t,A,n){var P,r,i,D,u,o;return P=t.i[1].a-t.i[0].a,r=t.i[1].b-t.i[0].b,D=(t.i[0].a+t.i[1].a)/2,u=(t.i[0].b+t.i[1].b)/2,P==0?r<0^n>u:r==0?P<0^A>D:(i=-P/r,o=(t.i[0].a+i*i*A-i*n+i*t.i[0].b)/(1+i*i),P<0^o>D);},At.FH=function(t,A,n,P,r){t.i[0].a=A,t.i[0].b=n,t.i[1].a=P,t.i[1].b=r;},At.GH=function(){this.i=j(l.fab,C1,55,2,0,1),this.i[0]=new Tt.vN(),this.i[1]=new Tt.vN(),this.a=0;},At.HH=function(t){var A,n,P,r;for(At.GH.call(this),A=0;A!=-1&&(n=w.aAb(t).indexOf('="',A),n!=-1);)if(P=(w.Wzb(A+1,n,w.aAb(t).length),w.aAb(t).substr(A+1,n-(A+1))),A=w.aAb(t).indexOf('"',n+2),r=A==-1?(w.Xzb(n+1,w.aAb(t).length+1),w.aAb(t).substr(n+1)):(w.Wzb(n+1,A,w.aAb(t).length),w.aAb(t).substr(n+1,A-(n+1))),T.kjb(P,"x1"))try{this.i[0].a=N0(r);}catch(i){if(i=M1(i),!Kt(i,40))throw Ut(i);}else if(T.kjb(P,"y1"))try{this.i[0].b=N0(r);}catch(i){if(i=M1(i),!Kt(i,40))throw Ut(i);}else if(T.kjb(P,"x2"))try{this.i[1].a=N0(r);}catch(i){if(i=M1(i),!Kt(i,40))throw Ut(i);}else if(T.kjb(P,"y2"))try{this.i[1].b=N0(r);}catch(i){if(i=M1(i),!Kt(i,40))throw Ut(i);}},N(61,193,{193:1,61:1,4:1},At.GH,At.HH),U.Eb=function(t,A){return this.a=At.CH(this,t,A),this.a!=0;},U.Fb=function(){this.a=0;},U.Gb=function(){var t;return t=new At.GH(),t.i[0].a=this.i[0].a,t.i[0].b=this.i[0].b,t.i[1].a=this.i[1].a,t.i[1].b=this.i[1].b,t.g=this.g,t;},U.Hb=function(t,A){return At.CH(this,t,A)!=0;},U.Ib=function(t,A){var n,P,r,i,D,u,o;C.$2(t,this.g?Lt.T3(W0,C.R2(t)):C.U2(t)),i=A?e.Ai(A,this.i[0].a):this.i[0].a,u=A?e.Bi(A,this.i[0].b):this.i[0].b,D=A?e.Ai(A,this.i[1].a):this.i[1].a,o=A?e.Bi(A,this.i[1].b):this.i[1].b,n=D-i,P=o-u,C.Z2(t,d.Math.max(1,.019999999552965164*d.Math.sqrt(n*n+P*P))),C.K2(t,i,u,D,o),r=new Tt.DN(4),Tt.yN(r,D+n/40,o+P/40),Tt.yN(r,D-n/5+P/10,o-P/5-n/10),Tt.yN(r,D-n/20,o-P/20),Tt.yN(r,D-n/5-P/10,o-P/5+n/10),C.P2(t,new C.n3(r));},U.Jb=function(t){var A,n,P,r,i,D;return D=At.DH(this),i=d.Math.max(D/8,3),A=e.no(this.i[0].a,this.i[0].b,this.i[1].a,this.i[1].b),P=d.Math.abs(i*d.Math.cos(A)),r=d.Math.abs(i*d.Math.sin(A)),n=new Tt.JN(),this.i[0].a<this.i[1].a?(n.c=this.i[0].a-P,n.b=this.i[1].a-this.i[0].a+2*P):(n.c=this.i[1].a-P,n.b=this.i[0].a-this.i[1].a+2*P),this.i[0].b<this.i[1].b?(n.d=this.i[0].b-r,n.a=this.i[1].b-this.i[0].b+2*r):(n.d=this.i[1].b-r,n.a=this.i[0].b-this.i[1].b+2*r),n;},U.Kb=function(){var t;return t=new T.Ujb(),T.Sjb(t,' x1="'+this.i[0].a+'"'),T.Sjb(t,' y1="'+this.i[0].b+'"'),T.Sjb(t,' x2="'+this.i[1].a+'"'),T.Sjb(t,' y2="'+this.i[1].b+'"'),t.a;},U.Lb=function(){return "arrow";},U.Mb=function(t){var A,n,P,r,i,D;switch(C.$2(t,C.W2(t)),this.a){case 1:C.O2(t,this.i[0].a-8,this.i[0].b-8,16);break;case 2:C.O2(t,this.i[1].a-8,this.i[1].b-8,16);break;case 3:i=At.DH(this),r=d.Math.max(i/8,3),A=e.no(this.i[0].a,this.i[0].b,this.i[1].a,this.i[1].b),n=r*d.Math.cos(A),P=-r*d.Math.sin(A),D=new Tt.DN(4),Tt.yN(D,this.i[0].a+n,this.i[0].b+P),Tt.yN(D,this.i[1].a+n,this.i[1].b+P),Tt.yN(D,this.i[1].a-n,this.i[1].b-P),Tt.yN(D,this.i[0].a-n,this.i[0].b-P),C.P2(t,new C.n3(D));}},U.Ob=function(t,A){switch(this.a){case 1:this.i[0].a=this.o[0]+t-this.k,this.i[0].b=this.p[0]+A-this.n;break;case 2:this.i[1].a=this.o[1]+t-this.k,this.i[1].b=this.p[1]+A-this.n;break;default:e.oe(this,t,A);}},U.a=0,l.A9=O(61),At.SH=function(t,A){var n,P,r,i,D,u,o,Q,L;return t==null?null:(u=t,Q=null,o=null,L=null,D=null,n=T.rjb(t,Y1(35)),n==-1?u=t:(u=(w.Wzb(0,n,w.aAb(t).length),w.aAb(t).substr(0,n)),P=T.sjb(t,Y1(35),n+1),P==-1?Q=(w.Xzb(n+1,w.aAb(t).length+1),w.aAb(t).substr(n+1)):(Q=(w.Wzb(n+1,P,w.aAb(t).length),w.aAb(t).substr(n+1,P-(n+1))),r=T.sjb(t,Y1(35),P+1),r==-1?o=(w.Xzb(P+1,w.aAb(t).length+1),w.aAb(t).substr(P+1)):(o=(w.Wzb(P+1,r,w.aAb(t).length),w.aAb(t).substr(P+1,r-(P+1))),i=T.sjb(t,Y1(35),r+1),i==-1?L=(w.Xzb(r+1,w.aAb(t).length+1),w.aAb(t).substr(r+1)):(L=(w.Wzb(r+1,i,w.aAb(t).length),w.aAb(t).substr(r+1,i-(r+1))),D=(w.Xzb(i+1,w.aAb(t).length+1),w.aAb(t).substr(i+1)))))),At.TH(u,(A&1)!=0?Q:null,(A&2)!=0?o:null,(A&4)!=0?L:null,(A&8)!=0?D:null,false,null));},At.TH=function(t,A,n,P,r,i,D){var u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V;if(t==null||w.aAb(t).length==0)return null;for(V=T.rjb(t,Y1(33)),c=false,s=V==0?1:0,p=0,o=0,D?At.fH(D):D=new At.zH();s!=-1;)V!=-1&&s>V&&(c=true),Q=T.sjb(t,Y1(32),s),!c&&(Q>V||Q==-1)&&(Q=V),L=null,Q==-1?(L=(w.Xzb(s,w.aAb(t).length+1),w.aAb(t).substr(s)),s=-1):(L=(w.Wzb(s,Q,w.aAb(t).length),w.aAb(t).substr(s,Q-s)),s=Q+1),M=null,A!=null&&w.aAb(A).length!=0&&(Q=T.sjb(A,Y1(32),p),Q==-1?M=(w.Xzb(p,w.aAb(A).length+1),w.aAb(A).substr(p)):(M=(w.Wzb(p,Q,w.aAb(A).length),w.aAb(A).substr(p,Q-p)),p=Q+1)),u=null,n!=null&&w.aAb(n).length!=0&&(Q=T.sjb(n,Y1(32),o),Q==-1?u=(w.Xzb(o,w.aAb(n).length+1),w.aAb(n).substr(o)):(u=(w.Wzb(o,Q,w.aAb(n).length),w.aAb(n).substr(o,Q-o)),o=Q+1)),S=new e._q(i),q=L==null?null:e.Sq(S,w.Czb((B=L,S1.Mmb(),B)),u==null?null:w.Czb((k=u,k)),0),M!=null&&e.Wq(S,w.Czb((g=M,S1.Mmb(),g))),c?(y.sj(D.f,q),D.d=-1):(y.sj(D.g,q),D.d=-1);if(P!=null&&w.aAb(P).length!=0&&At.vH(D,new e.hk(P)),r!=null&&w.aAb(r).length!=0){for(S=new e._q(i),h=0,b=T.rjb(r,Y1(43));b!=-1;)At._G(D,e.Pq(S,(w.Wzb(h,b,w.aAb(r).length),w.aAb(r).substr(h,b-h)))),h=b+1,b=T.sjb(r,Y1(43),h);At._G(D,e.Pq(S,(w.Xzb(h,w.aAb(r).length+1),w.aAb(r).substr(h))));}return D;},At.UH=function(t,A,n){var P,r,i,D,u,o,Q,L,s;return t==null?null:(o=t,L=null,Q=null,s=null,u=null,P=T.rjb(t,Y1(35)),P==-1?o=t:(o=(w.Wzb(0,P,w.aAb(t).length),w.aAb(t).substr(0,P)),r=T.sjb(t,Y1(35),P+1),r==-1?L=(w.Xzb(P+1,w.aAb(t).length+1),w.aAb(t).substr(P+1)):(L=(w.Wzb(P+1,r,w.aAb(t).length),w.aAb(t).substr(P+1,r-(P+1))),i=T.sjb(t,Y1(35),r+1),i==-1?Q=(w.Xzb(r+1,w.aAb(t).length+1),w.aAb(t).substr(r+1)):(Q=(w.Wzb(r+1,i,w.aAb(t).length),w.aAb(t).substr(r+1,i-(r+1))),D=T.sjb(t,Y1(35),i+1),D==-1?s=(w.Xzb(i+1,w.aAb(t).length+1),w.aAb(t).substr(i+1)):(s=(w.Wzb(i+1,D,w.aAb(t).length),w.aAb(t).substr(i+1,D-(i+1))),u=(w.Xzb(D+1,w.aAb(t).length+1),w.aAb(t).substr(D+1)))))),At.TH(o,L,Q,s,u,A,n));},At.VH=function(t,A,n){var P,r;return r=At.WH(t,A,(n&16)==0),r==null?null:(P=new T.Wjb(r[0]),n!=0&&(P.a+="#",(n&1)!=0&&r.length>1&&r[1]!=null&&T.Sjb(P,r[1])),n&=-2,n!=0&&(P.a+="#",(n&2)!=0&&r.length>2&&r[2]!=null&&T.Sjb(P,r[2])),n&=-3,n!=0&&(P.a+="#",(n&4)!=0&&r.length>3&&r[3]!=null&&T.Sjb(P,r[3])),n&=-5,n!=0&&(P.a+="#",(n&8)!=0&&r.length>4&&r[4]!=null&&T.Sjb(P,r[4])),P.a);},At.WH=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;for(L=j(l.Ybb,st,2,t.g.a.length+t.f.a.length,6,1),c=j(l.Ybb,st,2,t.g.a.length+t.f.a.length,6,1),i=j(l.Ybb,st,2,t.g.a.length+t.f.a.length,6,1),o=0;o<t.g.a.length+t.f.a.length;o++){if(k=o<t.g.a.length?y.xj(t.g,o):y.xj(t.f,o-t.g.a.length),k.K&&o<t.g.a.length)for(P=0;P<k.q;P++)e.vn(k,P,TA,false);if(r=new e.Ng(k),L[o]=e.rg(r),L[o]==null)return null;c[o]=e.og(r),i[o]=e.ng(r,A);}for(s=new T.Ujb(),D=new T.Ujb(),g=new T.Ujb(),Q=0;Q<t.g.a.length;Q++){if(h=Q,n)for(B="",h=-1,b=0;b<t.g.a.length;b++)T.ejb(B,L[b])<0&&(B=L[b],h=b);Q>0&&(s.a+=" ",g.a+=" ",D.a+=" "),T.Sjb(s,L[h]),T.Sjb(g,c[h]),T.Sjb(D,i[h]),L[h]="";}for(t.f.a.length!=0&&(s.a+="!",t.g.a.length!=0&&(g.a+=" ",D.a+=" ")),u=t.g.a.length;u<t.g.a.length+t.f.a.length;u++){if(h=u,n)for(B="",h=-1,b=t.g.a.length;b<t.g.a.length+t.f.a.length;b++)T.ejb(B,L[b])<0&&(B=L[b],h=b);u>t.g.a.length&&(s.a+=" ",g.a+=" ",D.a+=" "),T.Sjb(s,L[h]),T.Sjb(g,c[h]),T.Sjb(D,i[h]),L[h]="";}return M=j(l.Ybb,st,2,5,6,1),M[0]=s.a,w.aAb(g.a).length>t.g.a.length+t.f.a.length-1&&(M[1]=g.a),w.aAb(D.a).length>t.g.a.length+t.f.a.length-1&&(M[2]=D.a),t.b&&(M[3]=e.ek(t.b)),t.a.a.length!=0&&(M[4]=At.XH(t,A)),M;},At.XH=function(t,A){var n,P,r;for(r=new T.Ujb(),P=0;P<t.a.a.length;P++)w.aAb(r.a).length!=0&&(r.a+="+"),n=new e.Ng(y.xj(t.a,P)),T.Sjb(r,e.rg(n)),A&&(r.a+=" ",T.Sjb(r,e.ng(n,true)));return r.a;},At.YH=function(t,A,n,P,r){var i,D,u,o;if((A.u[n]&3)!=0&&(A.u[n]&4)==0&&H1(J(A.B[n],TA),0)&&(u=d.Math.abs(A.w[n]),u!=0)){for(D=0;D<t.c.g.a.length;D++)for(i=At.pH(t.c,D),o=0;o<i.f;o++)if(d.Math.abs(i.w[o])==u)return !!(A.j[n]==P.j[r]||pt(J(i.B[o],L0),0)||(i.u[o]&Q1)>>18!=(A.u[n]&Q1)>>18);}return  false;},At.ZH=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b;for(h=j(l.j7,H,6,t.f,15,1),b=At.aI(t,A,h),D=false,i=false,s=new y.Mj(),r=0;r<b;r++)e.ym(t,h[r])&&(Q=e.qw(t,h[r],P),(Q==1||Q==2)&&(u=e.Nl(n,P[h[r]]),(u==1||u==2)&&(Q==u?D=true:i=true)),o=j(l.j7,H,6,2,15,1),o[0]=h[r],o[1]=Q,w.szb(s.a,o));if(!D||!i)for(L=new y.Bqb(s);L.a<L.c.a.length;)Q=y.Aqb(L),o=Q[1],(o==1||o==2)&&i&&(o=o==1?2:1),e.un(n,P[Q[0]],o,true);},At._H=function(t){var A,n,P,r,i,D,u,o,Q,L,s;if(t.n==null){if(t.b)for(P=t.j+1,t.p=j(l.j7,c0,28,t.o.length,0,3),i=0;i<t.o.length;i++){for(t.p[i]=j(l.j7,E1,7,t.g[i].a.length,0,2),u=0;u<t.g[i].a.length;u++)t.p[i][u]=At.eI(t,i,y.xj(t.g[i],u),P);for(P+=t.o[i].f,r=At.pH(t.c,i),A=0;A<r.f;A++)pt(J(r.B[A],r0),0)||--P;}if(t.n=y1(l.j8,[c0,OA],[74,26],0,[t.q,t.c.f.a.length],2),t.d=y1(l.Ybb,[c0,st],[13,2],6,[t.q,t.c.f.a.length],2),t.e=j(l.Feb,ht,6,t.q,16,1),s=new e.Zv(),L=new T.Ujb(),D=0,n=0,t.q!=0){o=j(l.j7,H,6,t.o.length,15,1);do{for(Q=w.aAb(L.a).length,0<Q?L.a=T.Bjb(L.a,0,0):0>Q&&(L.a+=T.Hjb(j(l.g7,e0,6,-Q,15,1))),i=0;i<t.c.f.a.length;i++)$1(t.n[D],i,At.bI(t,t.g,o,i)),t.d[D][i]=e.rg(new e.Ng(t.n[D][i])),T.Sjb(L,t.d[D][i]);t.e[D]=e.Xv(s,L.a)==-1,(!t.t||!t.e[D])&&++n,++D;}while(At.hI(t,o)&&n<t.i);}}},At.aI=function(t,A,n){var P,r,i,D,u,o;for(u=j(l.Feb,ht,6,t.f,16,1),n[0]=A,o=1,u[A]=true,i=0;i<o;i++)for(D=0;D<e.$o(t,n[i]);D++)r=e._o(t,n[i],D),((t.F[r]&64)!=0||e.fm(t,r)==2||e.Vp(t,r))&&(P=e.Zo(t,n[i],D),u[P]||(n[o++]=P,u[P]=true));return o;},At.bI=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt,bt,xt,Et,qt,Zt,Nt,It,Bt,jt,b1,u1,Ht,gt,w1,x1,L1,O1,U1,l1,G1,a1,p0,J1,o1,D0,T0,d0,CA,jA,eA,nA,C0,MA,Z0,M0;for(Z=At.oH(t.c,P),o1=new e.tw(),g=0,B=0,Z0=null,yt=0;yt<t.o.length;yt++){for(z=At.pH(t.c,yt),e.bw(t.o[yt],1),U1=y.xj(A[yt],n[yt]),O1=j(l.j7,H,6,t.o[yt].f,15,1),p=j(l.Feb,ht,6,t.o[yt].f,16,1),q=j(l.Feb,ht,6,t.o[yt].g,16,1),Nt=0;Nt<z.f;Nt++)if(U1[Nt]!=-1)if(d.Math.abs(z.w[Nt])==0){for(S=U1[Nt],p[S]=true,w1=0;w1<e.$o(t.o[yt],S);w1++)q[e._o(t.o[yt],S,w1)]=true;for(gt=e.Lo(t.o[yt],S);gt<e.Mo(t.o[yt],S);gt++)q[e._o(t.o[yt],S,gt)]=true;}else O1[U1[Nt]]=d.Math.abs(z.w[Nt]);for(It=0;It<z.g;It++)if(s=z.D[0][It],h=z.D[1][It],d.Math.abs(z.w[s])!=0&&d.Math.abs(z.w[h])!=0&&(u=U1[s],o=U1[h],u!=-1&&o!=-1)){for(gt=0;gt<t.o[yt].g;gt++)if(e.Zl(t.o[yt],0,gt)==u&&e.Zl(t.o[yt],1,gt)==o||e.Zl(t.o[yt],0,gt)==o&&e.Zl(t.o[yt],1,gt)==u){q[gt]=true;break;}}for(a1=j(l.j7,H,6,t.o[yt].f,15,1),Bt=0;Bt<t.o[yt].f;Bt++)if(!p[Bt]&&(a1[Bt]=e.ll(t.o[yt],o1,Bt,g,B),t.b&&e.rn(o1,a1[Bt],t.p[yt][n[yt]][Bt],false),O1[Bt]!=0)){for(e.rn(o1,a1[Bt],O1[Bt],false),gt=0;gt<Z.f;gt++)if(d.Math.abs(Z.w[gt])==O1[Bt]){if(e.jn(o1,a1[Bt],Z.s[gt]),e.wn(o1,a1[Bt],Z.u[gt]&48),t.f[P][gt])if(J1=e.Nl(t.o[yt],Bt),T0=J(Z.B[gt],TA),et(T0,0)==0)e.un(o1,a1[Bt],3,false);else if(et(T0,TA)==0)!Z0&&(Z0=new y.Mj()),y.sj(Z0,T.Aib(a1[Bt]));else if(J1==1||J1==2){for(bt=et(T0,Bn)==0,x1=0;x1<e.$o(t.o[yt],Bt);x1++)if(b=e.Zo(t.o[yt],Bt,x1),p[b])for(L1=0;L1<e.$o(t.o[yt],Bt);L1++)p0=e.Zo(t.o[yt],Bt,L1),!p[p0]&&p0>b&&(bt=!bt);bt&&(J1=J1==1?2:1),e.un(o1,a1[Bt],J1,false);}else e.un(o1,a1[Bt],0,false);break;}}for(jt=0;jt<t.o[yt].g;jt++)q[jt]||e.nl(t.o[yt],o1,jt,g,B,a1,true);g=e.cn(o1,1),B=e.cn(o1,2);}for(G1=j(l.j7,H,6,Z.f,15,1),b1=0;b1<Z.f;b1++)if(O1=d.Math.abs(Z.w[b1]),O1==0)G1[b1]=e.ll(Z,o1,b1,g,B);else for(gt=0;gt<o1.q;gt++)if(d.Math.abs(o1.w[gt])==O1){G1[b1]=gt;break;}if(t.r)for(Nt=0;Nt<Z.f;Nt++)e.sn(o1,G1[Nt],true),e.zn(o1,G1[Nt],Z.J[Nt].a),e.An(o1,G1[Nt],Z.J[Nt].b);for(tt=0;tt<Z.g;tt++)if((Z.G[tt]&UA)==0&&(D0=e.ml(Z,o1,tt,g,B,G1[Z.D[0][tt]],G1[Z.D[1][tt]],false),d0=Z.G[tt]&31,d0!=0))for(I=false,e.Ln(o1,D0,31,false),ot=e.Ll(Z,Z.D[0][tt]),G=e.Ll(Z,Z.D[1][tt]),ct=0;!I&&ct<t.c.g.a.length;ct++)for(z=At.pH(t.c,ct),ut=0;!I&&ut<z.g;ut++)it=z.D[0][ut],Qt=z.D[1][ut],Dt=d.Math.abs(z.w[it]),lt=d.Math.abs(z.w[Qt]),Dt!=0&&lt!=0&&(Dt==ot&&lt==G||Dt==G&&lt==ot)&&(I=true,M0=z.G[ut]&31,M0!=0&&(U1=y.xj(A[ct],n[ct]),jA=U1[it],eA=U1[Qt],nA=e.Xo(t.o[ct],jA,eA),MA=e.im(t.o[ct],nA),C0=e.fm(t.o[ct],nA),M0&=7,d0&=7,M0==3&&d0==6?(o1.H[D0]=C0<=1?2:4,o1.T=0):M0==6&&d0==3?(o1.H[D0]=C0==3?2:1,o1.T=0):(o1.H[D0]=MA,o1.T=0)));for(e.bw(o1,1),k=false,u1=0;u1<Z.f;u1++)d.Math.abs(Z.w[u1])==0?(J1=e.qw(Z,u1,G1),e.un(o1,G1[u1],J1,false)):t.f[P][u1]&&At.YH(t,Z,u1,o1,G1[u1])&&(J1=e.qw(Z,u1,G1),e.un(o1,G1[u1],J1,false),(J1==1||J1==2)&&(M=(Z.u[u1]&Q1)>>18,c=(Z.u[u1]&Q1)>>18!=1&&(Z.u[u1]&Q1)>>18!=2?-1:(Z.u[u1]&F0)>>20,M==1?c+=g:M==2&&(c+=B),e.on(o1,G1[u1],M,c),k=true));for(k&&(g=e.cn(o1,1),B=e.cn(o1,2)),CA=j(l.Feb,ht,6,Z.q,16,1),Ht=0;Ht<Z.f;Ht++)t.f[P][Ht]&&!CA[Ht]&&(Z.u[Ht]&4)!=0&&H1(J(Z.B[Ht],TA),0)&&Z.j[Ht]==e.$o(o1,G1[Ht])&&At.ZH(Z,Ht,o1,G1);for(wt=j(l.Feb,ht,6,o1.q,16,1),Et=G1,qt=0,Zt=Et.length;qt<Zt;++qt)xt=Et[qt],wt[xt]=true;for(V=true;V;)for(V=false,L=0;L<o1.r;L++)u=o1.D[0][L],o=o1.D[1][L],wt[u]&&!wt[o]?(wt[o]=true,V=true):wt[o]&&!wt[u]&&(wt[u]=true,V=true);for(D=0;D<o1.q;D++)e.xn(o1,D,!wt[D]);if(e.wl(o1),!t.b)for(Q=0;Q<o1.q;Q++)o1.w[Q]=0;if(e.Ge(new e.Ve(o1),null,false),o1.T|=8,l1=2|(t.r?8:0),X.jA(new X.CA(l1),o1),Z0)for(e.bw(o1,15),i=new y.Bqb(Z0);i.a<i.c.a.length;)r=y.Aqb(i).a,(o1.u[r]&S0)!=0&&(e.un(o1,r,1,false),e.on(o1,r,1,g++));return o1;},At.cI=function(t){var A,n;if(A=t.q,t.t)for(n=0;n<t.q;n++)t.e[n]&&--A;return t.i>0&&A>t.i&&(A=t.i),A;},At.dI=function(t){var A,n,P;for(At._H(t),P=j(l.j8,c0,74,At.cI(t),0,2),A=0,n=0;n<t.q&&A<t.i;n++)(!t.t||!t.e[n])&&(P[A++]=t.n[n]);return P;},At.eI=function(t,A,n,P){var r,i,D,u;for(u=j(l.j7,H,6,t.o[A].f,15,1),y.Tqb(u,u.length,-1),i=At.pH(t.c,A),r=0;r<i.f;r++)n[r]!=-1&&(u[n[r]]=d.Math.abs(i.w[r]));for(D=0;D<u.length;D++)u[D]==-1&&(u[D]=P++);return u;},At.fI=function(t,A){var n,P,r,i;for(n=j(l.j7,H,6,t.j[A]-e.bp(t,A),15,1),i=0,r=0;r<t.j[A];r++)P=t.i[A][r],pt(J(t.B[P],r0),0)||(n[i++]=t.C[P]);return w.vzb(n,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),n;},At.gI=function(t,A){var n,P,r,i;for(i=j(l.j7,H,6,t.j[A]-e.bp(t,A),15,1),r=0,P=0;P<t.j[A];P++)n=t.i[A][P],pt(J(t.B[n],r0),0)||(i[r++]=d.Math.abs(t.w[n]));return w.vzb(i,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),i;},At.hI=function(t,A){var n;for(n=0;n<A.length;n++){if(A[n]<t.g[n].a.length-1)return ++A[n],true;A[n]=0;}return  false;},At.iI=function(t,A){var n,P,r,i,D,u,o,Q,L,s;for(t.j=0,r=0;r<A.g.a.length+A.f.a.length;r++)for(o=At.lH(t.c,r),n=0;n<o.f;n++)pt(J(o.B[n],r0),0)?o.w[n]=0:t.j<d.Math.abs(o.w[n])&&(t.j=d.Math.abs(o.w[n]));for(s=j(l.Feb,ht,6,t.j+1,16,1),i=0;i<A.g.a.length;i++)for(L=y.xj(A.g,i),n=0;n<L.f;n++)u=d.Math.abs(L.w[n]),u!=0&&(s[u]=true);for(D=0;D<A.f.a.length;D++)for(Q=y.xj(A.f,D),n=0;n<Q.f;n++)u=d.Math.abs(Q.w[n]),u!=0&&(s[u]?s[u]=false:Q.w[n]=0);for(P=0;P<A.g.a.length;P++)for(L=y.xj(A.g,P),n=0;n<L.f;n++)u=d.Math.abs(L.w[n]),u!=0&&s[u]&&(L.w[n]=0);},At.jI=function(t,A,n){var P,r,i,D,u,o,Q,L;if(t.o[A]=n,t.n=null,r=At.pH(t.c,A),e.bu(t.s,r,t.o[A]),o=8+(t.a?0:1),e.Pt(t.s,5,o)==0)return t.g[A]=new y.Mj(),t.q=0,false;for(t.g[A]=t.s.F,D=t.g[A].a.length-1;D>=0;D--)for(Q=y.xj(t.g[A],D),u=0;u<Q.length;u++)if(Q[u]!=-1&&(P=0,t.a&&r.s[u]==0&&e.Bl(t.o[A],Q[u])!=0&&(L=e.pm(t.o[A],Q[u]),P=e.mm(t.o[A],Q[u],L)),t.k[A][u]>0&&t.k[A][u]>e.jp(t.o[A],Q[u])-P)){y.Ej(t.g[A],D);break;}for(t.q=1,i=0;i<t.o.length;i++)t.q*=t.g[i]?t.g[i].a.length:0;return  true;},At.kI=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b;for(this.r=true,this.b=false,this.t=true,this.a=false,this.i=St,this.c=t,this.o=j(l.j8,OA,26,t.g.a.length,0,1),P=0;P<t.g.a.length;P++)e.On(y.xj(t.g,P),true),e.bw(y.xj(t.g,P),15);for(r=0;r<t.f.a.length;r++)e.bw(y.xj(t.f,r),15);for(At.iI(this,t),this.k=j(l.j7,E1,7,t.g.a.length,0,2),i=0;i<t.g.a.length;i++)for(s=y.xj(t.g,i),this.k[i]=j(l.j7,H,6,s.f,15,1),D=0;D<s.f;D++)if(Q=d.Math.abs(s.w[D]),Q!=0)for(u=0;u<t.f.a.length;u++)for(L=y.xj(t.f,u),o=0;o<L.f;o++)d.Math.abs(L.w[o])==Q&&(A=(b=e.pm(s,D),b+e.nm(s,D,b,s.C[D])-e.vp(s,D)-(h=e.pm(L,o),h+e.nm(L,o,h,L.C[o])-e.vp(L,o))),this.k[i][D]=d.Math.max(A,0));for(this.f=j(l.Feb,i1,12,t.f.a.length,0,2),n=0;n<t.f.a.length;n++)for(L=y.xj(t.f,n),this.f[n]=j(l.Feb,ht,6,L.f,16,1),D=0;D<L.f;D++)if(Q=d.Math.abs(L.w[D]),Q!=0)for(u=0;u<t.g.a.length;u++)for(s=y.xj(t.g,u),o=0;o<s.f;o++)d.Math.abs(s.w[o])==Q&&(Lt.$3(At.gI(L,D),At.gI(s,o))!=0||Lt.$3(At.fI(L,D),At.fI(s,o))!=0)&&(this.f[n][D]=true);this.s=new e.iu(),this.g=j(l.Fcb,i1,20,this.o.length,0,1);},At.lI=function(t){At.kI.call(this,t);},N(214,1,{},At.lI),U.a=false,U.b=false,U.i=0,U.j=0,U.q=0,U.r=false,U.t=false,l.C9=O(214),At.mI=function(t,A,n){var P,r,i;for(e.bw(A,1),r=y.xj(t.a,n),P=0;P<t.d.d.a.length;P++)i=Lt.f4(t.d,P),At.pI(i,A,r);e.vl(A);},At.nI=function(t,A){var n,P,r;if(t.f=A,e.bu(t.e,t.c,t.f),e.Pt(t.e,4,8)==0)return t.a=null,0;for(t.a=t.e.F,n=t.a.a.length-1;n>=0;n--)for(r=y.xj(t.a,n),P=0;P<r.length;P++)if(r[P]!=-1&&t.b[P]>0&&t.b[P]>e.jp(t.f,r[P])){y.Ej(t.a,n);break;}return t.a.a.length;},At.oI=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;for(e.bw(t,7),e.bw(A,7),this.c=t,this.d=new Lt.j4(),P=0;P<t.g;P++)Lt.d4(this.d,new At.sI(t,P));for(n=0;n<A.g;n++)L=new At.sI(A,n),o=(u=Lt.g4(this.d,L),u<0?-1:u),o==-1?(L.i=(At.BI(),At.xI),Lt.d4(this.d,L)):At.rI(Lt.f4(this.d,o),L);for(D=this.d.d.a.length-1;D>=0;D--)L=Lt.f4(this.d,D),L.i!=(At.BI(),At.AI)?Lt.f4(this.d,D).i==At.yI&&Lt.h4(this.d,D):L.i=At.zI;for(this.b=j(l.j7,H,6,t.f,15,1),i=0;i<t.f;i++)for(Q=0;Q<A.f;Q++)if(d.Math.abs(A.w[Q])==d.Math.abs(t.w[i])){r=(h=e.pm(t,i),h+e.nm(t,i,h,t.C[i])-e.vp(t,i)-(s=e.pm(A,Q),s+e.nm(A,Q,s,A.C[Q])-e.vp(A,Q))),this.b[i]=r>0?r:0;break;}this.e=new e.iu();},N(285,1,{},At.oI),l.F9=O(285),At.pI=function(t,A,n){var P,r,i,D;if(P=n[t.a],r=n[t.b],t.i==(At.BI(),At.xI))return e.Wk(A,P,r,t.g),true;if(i=e.Xo(A,P,r),t.i==At.vI)return e.Mn(A,i,t.g),true;if(t.i==At.wI){if(D=e.fm(A,i)+t.f,D>=0&&D<=3)return e.Mn(A,i,(e.Rk(),D==0?16:D==1?1:D==2?386:D==3?4:D==4?32:64)),true;}else if(t.i==At.zI)return A.H[i]=512,true;return  false;},At.qI=function(t,A){return t.d!=A.d?t.d<A.d?-1:1:t.e!=A.e?t.e<A.e?-1:1:0;},At.rI=function(t,A){var n,P;t.c==A.c?t.i=(At.BI(),At.yI):Pe(A.c)==1?(t.g=A.c==1?1:A.c==2?386:A.c==4?4:A.c==16?16:8,t.i=(At.BI(),At.vI)):(n=(t.c&16)!=0?0:(t.c&1)!=0?1:(t.c&2)!=0?2:3,P=(A.c&16)!=0?0:(A.c&1)!=0?1:(A.c&2)!=0?2:3,P==n?t.i=(At.BI(),At.yI):(t.f=P-n,t.i=(At.BI(),At.wI)));},At.sI=function(t,A){var n,P,r;this.a=t.D[0][A],this.b=t.D[1][A],P=e.Ll(t,this.a),r=e.Ll(t,this.b),P<r?(this.d=P,this.e=r):(this.d=r,this.e=P),this.c=t.G[A]&31,n=t.H[A]&127,n==16?this.c|=16:n==2?this.c|=2:n==4?this.c|=4:this.c|=1,this.i=(At.BI(),At.AI);},N(183,1,{183:1,32:1},At.sI),U.Sb=function(t){return At.qI(this,t);},U.mb=function(t){return this.d==t.d&&this.e==t.e;},U.a=0,U.b=0,U.c=0,U.d=0,U.e=0,U.f=0,U.g=0,l.E9=O(183),At.BI=function(){At.BI=D1,At.xI=new At.CI("CREATE",0),At.wI=new At.CI("CHANGE_DIF",1),At.vI=new At.CI("CHANGE_ABS",2),At.zI=new At.CI("REMOVE",3),At.yI=new At.CI("NO_CHANGE",4),At.AI=new At.CI("UNKNOWN",5);},At.CI=function(t,A){vn.call(this,t,A);},At.DI=function(){return At.BI(),v(F(l.D9,1),s0,98,0,[At.xI,At.wI,At.vI,At.zI,At.yI,At.AI]);},N(98,56,{98:1,4:1,32:1,56:1},At.CI),l.D9=GA(98,At.DI),a.EI=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(e.bw(A,1),c=t.k,g=0,B=c.length;g<B;++g)b=c[g],Q=n[b.a],L=n[b.b],s=e.Xo(A,Q,L),s==-1?e.Wk(A,Q,L,b.e):b.e==-1?A.H[s]=512:b.e!=-2&&e.Mn(A,s,b.e);if(u=e.vl(A),t.a.length!=0)for(e.bw(A,7),r=t.a,i=0,D=r.length;i<D;++i)P=r[i],o=u==null?n[P]:u[n[P]],h=A.u[o]&3,e.un(A,o,h==1?2:1,false),e.wq(A,o);return u;},a.FI=function(t){var A,n,P,r,i;for(i=new a.bJ(t.g,t.f),r=j(l.j7,H,6,t.g.q,15,1),n=0;n<t.g.q;n++)r[n]=e.Ll(t.g,n);for(P=j(l.j7,H,6,t.f.q,15,1),A=0;A<t.f.q;A++)P[A]=e.Ll(t.f,A);return -a.aJ(i,a.VI(i,r,P))*.25;},a.GI=function(t,A){var n,P,r,i,D;for(t.i=j(l.j7,H,6,t.g.f,15,1),P=0;P<t.f.f;P++)e.Ll(t.f,P)!=0&&(t.i[A[e.Ll(t.f,P)]]=e.mw(t.f,P));if(D=j(l.j7,H,6,t.g.q,15,1),i=e.hp(t.g,D,false,false),i>1)for(r=j(l.j7,H,6,i,15,1),n=0;n<t.g.f;n++)t.i[n]|=r[D[n]]++<<5;},a.HI=function(t,A,n,P,r){var i,D,u,o;if(t.j.length==0||t.n.a.length==0)return null;for(o=j(l.Feb,i1,12,A,0,2),u=j(l.Feb,ht,6,P,16,1),D=0;D<r.length;D++)r[D]!=-1&&e.Ll(t.f,D)==0&&(u[r[D]]=true);for(i=0;i<n.length;i++)n[i]!=-1&&e.Ll(t.g,i)==0&&(o[n[i]]=u);return o;},a.II=function(t,A,n){var P,r,i,D,u,o,Q;if(o=false,e.Qo(t.g,A)==0)for(u=1;u<e.$o(t.g,A);u++)for(Q=0;Q<u;Q++)P=e.Zo(t.g,A,u),r=e.Zo(t.g,A,Q),i=e.Ll(t.g,P),D=e.Ll(t.g,r),n[i]>n[D]^P>r&&(o=!o);return o;},a.JI=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z;for(this.d=t,z=At.UH(A,false,null),(z.g.a.length!=1||z.f.a.length!=1)&&T.Zjb(),z.c||At.gH(z)||T.Zjb(),this.g=y.xj(z.g,0),this.f=y.xj(z.f,0),e.bw(this.g,1),e.bw(this.f,63),Q=new Lt.j4(),this.c=j(l.j7,H,6,this.g.f+1,15,1),this.c[0]=-1,P=0;P<this.g.f;P++)e.Ll(this.g,P)!=0&&(this.c[e.Ll(this.g,P)]=P);for(a.GI(this,this.c),V=j(l.Feb,ht,6,this.g.g,16,1),k=0;k<this.f.g;k++)g=e.Zl(this.f,0,k),B=e.Zl(this.f,1,k),h=e.Ll(this.f,g),b=e.Ll(this.f,B),h!=0&&b!=0&&(r=this.c[h],i=this.c[b],M=e.jm(this.f,k),S=e.Xo(this.g,r,i),S==-1?Lt.d4(Q,new a.LI(r,i,h,b,M)):((e.hm(this.g,S)&31)==0&&(I=e.jm(this.g,S),I!=M&&Lt.d4(Q,new a.LI(r,i,h,b,M))),V[S]=true));for(o=0;o<this.g.g;o++)V[o]||(r=e.Zl(this.g,0,o),i=e.Zl(this.g,1,o),!e.Jm(this.g,r)&&!e.Jm(this.g,i)&&(h=e.Ll(this.g,r),b=e.Ll(this.g,i),(h!=0||b!=0)&&Lt.d4(Q,new a.LI(r,i,h,b,-1))));for(this.k=Lt.i4(Q,j(l.G9,{463:1,4:1,5:1,11:1,8:1},148,0,0,1)),L=0,D=0;D<this.g.f;D++)e.Ll(this.g,D)==0&&!e.Jm(this.g,D)&&++L;for(this.j=j(l.j7,H,6,L,15,1),L=0,u=0;u<this.g.f;u++)e.Ll(this.g,u)==0&&!e.Jm(this.g,u)&&(this.j[L++]=u);for(this.n=new y.Mj(),s=j(l.Feb,ht,6,this.f.f,16,1),n=0;n<this.f.f;n++)e.Ll(this.f,n)==0&&!e.Jm(this.f,n)&&!s[n]&&y.sj(this.n,new a.VJ(this.f,n,s));for(this.b=this.j.length==0&&this.n.a.length==0,this.a=j(l.j7,H,6,0,15,1),c=0;c<this.f.f;c++)p=e.Nl(this.f,c),(p==1||p==2)&&(q=this.c[e.Ll(this.f,c)],q!=-1&&(Z=e.Nl(this.g,q),a.II(this,c,this.c)==(Z==p)&&(this.a=y.Jqb(this.a,this.a.length+1),this.a[this.a.length-1]=q)));this.e=a.FI(this);},N(199,1,{199:1},a.JI),U.b=false,U.e=0,l.I9=O(199),a.KI=function(t,A){return t.c<A.c?-1:t.c>A.c?1:t.d<A.d?-1:t.d>A.d?1:0;},a.LI=function(t,A,n,P,r){n<P?(this.a=t,this.b=A,this.c=n,this.d=P):(this.a=A,this.b=t,this.c=P,this.d=n),this.e=r;},N(148,1,{148:1,32:1},a.LI),U.Sb=function(t){return a.KI(this,t);},U.a=0,U.b=0,U.c=0,U.d=0,U.e=0,l.G9=O(148),a.QI=function(){a.QI=D1,a.NI=v(F(l.Ybb,2),c0,13,0,[v(F(l.Ybb,1),st,2,6,["e","daXJ@@PjyITuiX@`!dahJ@@SJYIMMfPB#IaLJfxP IaAhl[`#!BDpAl@AL@[@Be}aL@[@@ !B|Osp?QZR?O_}}zGze`@##"]),v(F(l.Ybb,1),st,2,6,["g","gBa`@lde@XS@!gCA`@mDPOdku`#qbq qJt#!B[G|S_qgq !BjW}q]cga##"]),v(F(l.Ybb,1),st,2,6,["o","gGQHDDqIUjdB!gGQHHDqAeMjhB#qbqk qfQ]#!B@AL@[@@S@Fp@Dp !BFQ{~_q|ZGvUSYp##"]),v(F(l.Ybb,1),st,2,6,["r","gOQdEjHbAFQRBrtAaJ!gNQdEbHbABCTKPFDH#qbMwX qbM~X#!BNDm`ra?UjW~YBYX@ !Ba[zw?_x@?g~H?XO~##"]),v(F(l.Ybb,1),st,2,6,["s","gOp`ATigujj`H!gOp`ATiVKjj`H#qnyS` qY~eP#!BTLtk^sE?BOs|]pc} !BbOvw?_y??g~H?[_}##"]),v(F(l.Ybb,1),st,2,6,["t","gOP`@dcUZdB!gNp`@tiTMjj@`#q{ir` qkLrx#!Be`Bzr_wp?OC}|Osp !B?g~w@k_}m?vw@n[a##"]),v(F(l.Ybb,1),st,2,6,["u","daXB`Hrn@HrPEILt`D!daXB`DJn@HRUMjV@H#IxVaLJ` IylJhPP#!B[G}l@OKyDpAl@AL@[@@ !B@Fp@DpAl@AN]?`Al@AL##"]),v(F(l.Ybb,1),st,2,6,["Sakurai","gOQH@wAINvZ@pdcFe@x@!gOQH@wAIgJi@pdcFe@x@#qreKx qrLkx#!BDpAl@IknDw|S@Fp@ !Bb@JH?_x@b@JH?Ven##"]),v(F(l.Ybb,1),st,2,6,["Mitsunobu","gFP`ATfRhdPp`}KEYg]d@!gFP`ATfRhd`pekL{l`#qrLk qZLn#!B@hc}b@C~@h`YM` !B@hc}b@C~@h`YM`##"]),v(F(l.Ybb,1),st,2,6,["Aldol-Addition","gOQ@AdTAcS@[Q^crJTLES`DJsL?vH!gGQ@@dsuRAcJg@HUaX#qYEbp qYub#!BxOyBzLKg`dG~xG~{ !Bb@K~@Hc}FBIA@@##"]),v(F(l.Ybb,1),st,2,6,["Aldol-Condensation","gOQ@AdTAcS@[Q^crJTLES`DJsL?vH!gFQ@@`rrpdlHHpipBEXf@#qYEbp q^aU#!B{ZRRqA?AQfyA@L_C !B}QFw@h`B_tnH_P##"]),v(F(l.Ybb,1),st,2,6,["Acetal-Aldol-Addition","dmdB@@serQS@sJjfd@p`Xir\\@`j\\@aUJXK@!daxL@@[df[ZjT@qQdxACdxABjTqf@#qB@`OuX qBtM{#!B_]]}rHKWw?y}uy[~GJbBu{wWqG| !BfJK~TIa]fJJghg{`pP@##"]),v(F(l.Ybb,1),st,2,6,["Acetal-Aldol-Condensation","dmdB@@serQS@sJjfd@p`Xir\\@`j\\@aUJXK@!gNp`CTjUiV@qQS`DJg@HUVXV@#qB@`OuX qqj{`#!B?[_}b@Jw_?{}m~[~[N@Bm?vwkN| !BfsK~yzPrw}m?rzQM##"]),v(F(l.Ybb,1),st,2,6,["Acetal-Aldol-Condensation-Cyclization","dkLB@@ZURYUvUjljh@paHpr\\@`!didD@@yIfUXXHL@CFNS`D@#IXljNPY@@@ IXljIyA#!BbOw~_x`Bm?vH?[_}b@JH?_y?b@Jw?Xc} !BbOvH?Oy??`BH?Xa}?`C~_p##"]),v(F(l.Ybb,1),st,2,6,["Enolester-Cleavage","gOQ`@fdscT`_Qp!gOQ`@cdTASS@P#q}Kr` q}cNP#!B@k]}mpC~@k]}mqdY !Bb@K~@Hc}BfzH@hc}##"]),v(F(l.Ybb,1),st,2,6,["propargylEnone","gCa@@dmXFD@!gCa@@dkHD#qNT qLV#!BbOw~_?y? !B@AL@[@@S##"]),v(F(l.Ybb,1),st,2,6,["Arndt-Eistert","daiDaJYBBHj^{HhAYMpAaA@!daiD`FzLBHPVsZl@p`@#IyHHZ[@ IzDGBi`#!B?`Bw?H`Bn]{~\\g?~@Ox !B@rzH?_y?b@JH?_n~bOt##"]),v(F(l.Ybb,1),st,2,6,["Curtius","gO]IcVaDF[s{HhCIe@`!gN]HMWADHJfm`XP@#q~Jk` qytUX#!B@O{|b@Jw\\o{~_?x@ !Bj]y??g?~?[^G_hc}##"]),v(F(l.Ybb,1),st,2,6,["diazomethanHomologation","gFU@lQioIIs\\AyH!gFU@CPdimXD#qbM^ qbqk#!B?X`BbFZN?`C~_p !B@AL@[@@Su[x@Dp##"]),v(F(l.Ybb,1),st,2,6,[wD,"deD@@LdbEdSTu@FqHWSDda`JFChXIa`?tdKi@!deD@@Ldb`\\SUM@FqHPsDda`JF@XXIa`?tdHY@#qTEApX qQECf@#!BQzK~}ubbW`BEgcb]?a@gg[zO !BQzK~}ubbW`Ag{VVAQzJ~c?xP##"]),v(F(l.Ybb,1),st,2,6,[wD,"deD@@LdbEdSTu@FqHWSDda`JFChXIa`?tdKi@!gC`@DiZDE@#qPDA@p qQf#!BmpK~_x`Bm?tAs[]}?`BH_[_| !B_vp@[G|S##"]),v(F(l.Ybb,1),st,2,6,["yne-Metathesis","daX@@LdPLWWPAlRXwQIHXLa`ZFChXO}IL[`!daX@@LdPLWWPAlRXwQIHXLa`ZFChXO}IL[`#qT@q@ qt@Q@#!BZmoi@Fjo|SFe|IkGiU@ !BZmoi@Fjo|SFe|IkGiU@##"]),v(F(l.Ybb,1),st,2,6,["ene-yne-Metathesis","dcd@@LdPLPLWSSPAlrXwKlVRFCHXFa`zFAXXMa`?udqnWP!dcT@@LdbbplTsTt@[MeMr{Ela`jFAhXNa`VFCXXO}[J[et#qe@N@S@ qeHP@s@#!B?g?~@Oy?^gG}bOvw?H`E@PJw@hc}mp !B?`BH?[_}mpJH@oy??`AuC`Jw@hc}mp##"]),v(F(l.Ybb,1),st,2,6,["Alkyne-Cyclisation","gG`@DcO|CFAFC`@!gFp@DiTt@@CFAFC`@#qi\\V qiSt#!B_y[qA`Biu^zV@@ !B?g~w@k_}m?vw@`##"]),v(F(l.Ybb,1),st,2,6,["Elimination-Claisen","gNp`AldTQji@~a`!gGQ@@djmRA@#qtHUX qtSi#!Bm?vH?[\\B?g~H@hc} !B@AL@[@@S@Fp@Dp##"]),v(F(l.Ybb,1),st,2,6,["imineFormationAzaCope","daZH@LAIMUjd@pRL@!daZH@HAAn]jd@p`@#IGfaLJ` IFDzfK@#!BDpAl@IkqDpAl@AL@[@@ !BFaFw@h`BbOw~@H`BbOt##"]),v(F(l.Ybb,1),st,2,6,["didehydroCopeWithAromatisation","gNp@Di]ejDcjcr|wK`!gOp@DjWkB@@H#qrLkx q\\oQp#!B?g~H?K_}bGvH?H`B !Bm?w~@Hc}mpJw@ox@##"]),v(F(l.Ybb,1),st,2,6,["symAldolNitrogenRing","dovJ@GBfttf\\v\\qjViPCADGbDodnGp!doNJ@JCSmtefWTCaYjje@H#IlZXi]]yL~C IqMVCzaIim?#!BQtl_riY?Qtl_rfuvNCQ`uZd@NCQ`uVVu}?sA]P !B?`BH@ox@bGvH@k\\Bb@JH_Xa}b@K~_rYltUr|W@##"]),v(F(l.Ybb,1),st,2,6,["Diels-Alder","gFP@LdPLjA@!gFp@DiTujhB#qiZf qMSf#!B?_C}}?spIPFV@@ !B?g~w@k_}m?vw@`##"]),v(F(l.Ybb,1),st,2,6,["Claisen-RA","gGQ@@eKtRA@!gGQ@@djqRA@#qMsT qM\\V#!B_qL@Dw}l@Fp@Dp !B@AL@[@@S@Fp@Dp##"]),v(F(l.Ybb,1),st,2,6,["Cope","gGQ@DeZmRAbhcApIF@P@!gGQ@HeZmRAbhc@HIFC`@#qkNT qi\\V#!B_vp@[@@S@Fp@Dp !B_vp@[@@S@Fp@Dp##"]),v(F(l.Ybb,1),st,2,6,["OxyCope","gNq@@dr}SHFD@!gNq@@djkUHD#qynZ` qykbp#!B?g~w?Xc}mpJH@hc} !B@Fp@DpAl@AL@[@@S##"]),v(F(l.Ybb,1),st,2,6,["Vinylcyclopropane","gKP@DklVj@`!gKP@Di\\Vj@`#qaVh qTqh#!Bm?vH?PC~?K\\ !B?g|_Fa}eTv\\##"]),v(F(l.Ybb,1),st,2,6,["Furan-CH2-OH-Rearrangement","gOp`@tiguif`H!gOp`@tfWMZZ`H#qZna@ qZtM@#!BTLtk^sE?BOs|]pc} !BBOuPtdy?UGm@V]Ho##"]),v(F(l.Ybb,1),st,2,6,["rearrangement1032","gOp`ATieMji`H!gOp`ATeekZj`H#qaSnx qa]~P#!BTLtk^pc|LW?|]pc} !BBOpH?UCRSg}T^tAY##"]),v(F(l.Ybb,1),st,2,6,["Pinacol-Rearrangement","gNp`ATffjj@pPh!gOP`@tfXZhCAA`#qb^ix qb^oH#!B@k^H@k_}@k_~@Hc} !BbOvH@oy??`BH?PFf##"]),v(F(l.Ybb,1),st,2,6,["1,3-WaterEliminationWith12Shift","gJQ@@dmU@_SP!gKa@@`vtA}A@#qiTp qjap#!BbOvH@ox@bOt !BJdE?[@Al@AL##"]),v(F(l.Ybb,1),st,2,6,["Epoxydation","gB``ADcdCB@!gC``AhtUPGtt@#qqb qtQ#!BjW}Y\\YX@ !B?g~w?^Va##"]),v(F(l.Ybb,1),st,2,6,["oxydativePropargylAmine13Shift","gKi@HDEZpLHOQP!gJY@BDeVXB#qMr` qNTh#!BqiXTy{U?mW| !B@Fp@DpAl@AL##"]),v(F(l.Ybb,1),st,2,6,["Baeyer-Villiger","gFQ`@[dTAZ`LHP!gFQ`@jdrMPGtl@#qrak qrlK#!B_?{}mwvHs^FVP@ !BbOvH@oy?bOuQzP##"]),v(F(l.Ybb,1),st,2,6,[gD,"gGa@@dYs@XHFJIBDQbUeHPbLRl@!gFQ@@eNUPFJIBDQbUeHPbLRls@`#qbq] qfQk#!B@AOIDW}l@tD@Dp !B_qL@Dw}l_qNcDP##"]),v(F(l.Ybb,1),st,2,6,[gD,"gBa@@d\\`XP@!gJQ@@eOU@XpdHQFIVY`P#qbq qfQ@#!B@AOIDW}l !B_qL@Dw}l_qL##"]),v(F(l.Ybb,1),st,2,6,["Hantzsch Thiazol","daZHPDp@chaIMefh@ppDzTD~hYmC^bhbcPp]dQbUg~pp!gKXHL@aJWFe`H#qNPe@ qNj`#!BvuK[KUxv_yS[k_zhvuH !BTqa`FbpX?`@##"]),v(F(l.Ybb,1),st,2,6,["Oxadiazole","gOX`BEdTASW@XQ@!gOu@HPeKNMKTA@#qrDMX qpULX#!BmpK~@K_}Mlx@?`C~ !BZ?`C}v|m_rYR[z?\\##"]),v(F(l.Ybb,1),st,2,6,["Imidazole","dmeHPNg@qJqLbTtATijZ@LLJnuDmhWtSDXUFC`?rIoTAP!gOt@ATieuej`H#qDPpM_@ q~ZM`#!BqvKGg_yOqvKGg_xphrGkLcz@_sD !BTMHkACD@BOw|B@QT##"]),v(F(l.Ybb,1),st,2,6,["1,2,3-Triazole","gB`HAbIGXFDWiM@!gF|@ADeKXmT`P#QIp Q@v#!B_vpU?g}l !BTv]`YRqg?g|XK@##"]),v(F(l.Ybb,1),st,2,6,["1,2,4-Triazole","deFD@NALbbfASUW@FD]YJZLUCAVJ}?nES@!gO|@ABeKNLuRA@#qDB@FM q@LuP#!BY?r~@F_]jDJW`j`}Iaxx[UC] !BTv]@IPqgog|hCBT_##"])]);},a.RI=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it;for(a.UI(),o=new a.kK(),a.hK(o,A),k=o.j,h=o.d,e.bw(k,1),e.bw(h,1),S=new e.iu(),B=new e.iu(),e.cu(S,k),e.cu(B,h),t.c=P1,D=null,i=null,r=0,tt=0,t.a=new T.Ujb(),n=new e.tw(),I=a.OI,Z=0,z=I.length;Z<z&&(V=I[Z],tt++!=t.b);++Z)if(e._t(S,V.g),e.au(S,V.i),e.Pt(S,6,8)!=0&&(e._t(B,V.f),e.Pt(B,2,8)!=0&&S.F.a.length<=512)){for(u=-1e4,g=y.xj(B.F,0),q=new y.Bqb(S.F);q.a<q.c.a.length&&(p=y.Aqb(q),!(tt++>=t.b));)e.ol(k,n),Q=a.EI(V,n,p),it=a.HI(V,k.f,p,h.f,g),e.bw(n,1),P=j(l.j7,H,6,n.f,15,1),c=j(l.j7,H,6,h.f,15,1),a.eK(o,n,h,P,c,it),G=o.q-V.e,u<G&&(u=G),t.c<G&&(t.c=G,D=a.SI(P,c,Q,k.f),i=c,r=o.b);s=o.a<=1?"":jD+o.a+")",T.Sjb(t.a,V.d+(V.b?"_S":V.j.length==0?"_L":V.n.a.length==0?"_I":"_U")+(Lt.W3(),Lt.X3(u,5))+s+k1);}M=j(l.j7,H,6,k.f,15,1),b=j(l.j7,H,6,h.f,15,1),a.eK(o,k,h,M,b,null),ot=o.q,t.c<=ot&&(t.c=ot,D=M,i=b,r=o.b),L=o.a<=1?"":jD+o.a+")",T.Sjb(t.a,(Lt.W3(),"no rule:"+Lt.X3(ot,5)+L+k1)),t.c!=P1&&a.YJ(o,A,D,i,r);},a.SI=function(t,A,n,P){var r,i,D,u,o;for(D=j(l.Feb,ht,6,t.length+1,16,1),o=j(l.j7,H,6,P,15,1),u=0;u<P;u++)r=n==null?u:n[u],r!=-1&&(o[u]=t[r],D[o[u]]=true);for(i=0;i<A.length;i++)D[A[i]]||(A[i]=0);return o;},a.TI=function(){a.QI(),this.b=St;},a.UI=function(){var t,A;if(a.OI==null&&a.OI==null){for(t=j(l.I9,C1,199,a.NI.length,0,1),A=0;A<a.NI.length;A++)t[A]=new a.JI(a.NI[A][0],a.NI[A][1]);a.OI=t;}},N(294,1,{},a.TI),U.b=0,U.c=0,l.H9=O(294),a.VI=function(t,A,n){var P,r,i,D;for(i=j(l.j7,H,6,t.b.f+1,15,1),y.Tqb(i,i.length,-1),r=0;r<n.length;r++)n[r]!=0&&(i[n[r]]=r);for(D=j(l.j7,H,6,t.b.f,15,1),y.Tqb(D,D.length,-1),P=0;P<A.length;P++)D[P]=i[A[P]];return D;},a.WI=function(t,A){var n,P,r,i;return n=t.D[0][A],P=t.D[1][A],r=e.Hm(t,n),i=e.Hm(t,P),!r&&!i?A<t.g&&e.pt(t.p,A)?Ii:mr+e.fm(t,A)/10:r&&i||r&&e.Lm(t,P)||i&&e.Lm(t,n)?bn:r&&a.oK(t,P,n)||i&&a.oK(t,n,P)?oP:r&&a.mK(t,P,n)||i&&a.mK(t,n,P)?Hr:r&&P<t.f&&e.ot(t.p,P)||i&&n<t.f&&e.ot(t.p,n)?1.9500000476837158:mr;},a.XI=function(t,A){return A<t.g&&(e.ut(t.p,A)||t.H[A]==8)?0:t.H[A]&127;},a.YI=function(t,A){return A<t.g&&(e.ut(t.p,A)||t.H[A]==8)?1.5:e.fm(t,A);},a.ZI=function(t,A,n){var P,r,i;if(P=n[A],P!=-1&&e.Nl(t.a,P)!=0&&a.$I(t,A,P,n)){if(i=e.Nl(t.b,A),r=e.Nl(t.a,P),i==3){if(r==1||r==2)return 5;}else if(r==3||a._I(t,A,n)==(i==r))return 5;}return 0;},a.$I=function(t,A,n,P){var r,i,D,u;if(e.$o(t.b,A)!=e.$o(t.a,n))return  false;for(i=0;i<e.$o(t.b,A);i++){for(u=e.Zo(t.b,A,i),r=false,D=0;D<e.$o(t.a,n);D++)if(P[u]==e.Zo(t.a,n,D)){r=true;break;}if(!r)return  false;}return  true;},a._I=function(t,A,n){var P,r,i,D,u,o,Q;if(o=false,e.Qo(t.b,A)==0)for(u=1;u<e.$o(t.b,A);u++)for(Q=0;Q<u;Q++)P=e.Zo(t.b,A,u),r=e.Zo(t.b,A,Q),i=n[P],D=n[r],i!=-1&&D!=-1&&i>D^P>r&&(o=!o);return o;},a.aJ=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(s=0,D=j(l.Feb,ht,6,t.a.f,16,1),P=A,r=0,i=P.length;r<i;++r)n=P[r],n!=-1&&(D[n]=true);for(h=j(l.Feb,ht,6,t.a.g,16,1),B=0;B<t.b.g;B++)if(c=e.Zl(t.b,0,B),g=e.Zl(t.b,1,B),!(t.b.K&&(e.Jm(t.b,c)||e.Jm(t.b,g)))){if(u=A[c],o=A[g],a.YI(t.b,B),u==-1||o==-1){(u!=-1||o!=-1)&&(s+=a.WI(t.b,B));continue;}if(L=e.Xo(t.a,u,o),L==-1){s+=a.WI(t.b,B);continue;}h[L]=true,s+=a.XI(t.b,B)==a.XI(t.a,L)?0:1;}for(Q=0;Q<t.a.g;Q++)t.a.K&&(e.Jm(t.a,e.Zl(t.a,0,Q))||e.Jm(t.a,e.Zl(t.a,1,Q)))||h[Q]||(s+=a.WI(t.a,Q));for(b=0;b<t.b.f;b++)e.Nl(t.b,b)!=0&&(s+=a.ZI(t,b,A));return -s;},a.bJ=function(t,A){this.b=t,this.a=A;},N(202,1,{},a.bJ),l.J9=O(202),a.cJ=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;if(k=new a.bJ(t.e,t.c),g=a.VI(k,t.f,t.d),t.a.a.length==0)return a.aJ(k,g);for(M=1,q=new y.Bqb(t.a);q.a<q.c.a.length;)p=y.Aqb(q),M*=a.iJ(p);if(M>4e6)return T.Zjb(),-M;for(A=0,D=j(l.j7,H,6,t.a.a.length,15,1),b=j(l.j7,H,6,t.a.a.length,15,1),o=0;o<t.a.a.length;o++)p=y.xj(t.a,o),b[o]=a.lJ(p),D[o]=A,A+=p.a;for(i=xr,r=null,P=null,c=j(l.j7,H,6,t.a.a.length,15,1),h=t.a.a.length!=0;h;){if(S=false,t.i!=null){for(Q=0;Q<t.a.a.length;Q++)if(a.mJ(y.xj(t.a,Q),c[Q],t.i)){S=true;break;}}if(!S){for(Q=0;Q<t.a.a.length;Q++)a.hJ(y.xj(t.a,Q),c[Q],g);if(B=a.aJ(k,g),i<B)for(i=B,r=j(l.j7,H,6,A,15,1),P=j(l.j7,H,6,A,15,1),n=0,L=0;L<t.a.a.length;L++)p=y.xj(t.a,L),a.kJ(p,c[L],r,n),a.jJ(p,c[L],P,n),n+=p.a;}for(h=false,u=0;u<c.length;u++){if(++c[u],c[u]<b[u]){h=true;break;}c[u]=0;}}if(i!=xr)for(s=t.b,u=0;u<A;u++)++s,t.f[r[u]]=s,t.d[P[u]]=s;return i;},a.dJ=function(t,A,n,P,r,i){var D,u,o,Q,L,s,h,b;for(this.e=t,this.c=A,this.f=n,this.d=P,this.g=r,this.b=r,this.i=i,Q=new y.fwb(),u=0;u<t.f;u++)n[u]==0&&(o=t.C[u]+(t.A[u]<<16),h=y.opb(Q,T.Aib(o)),h||(h=new a.pJ(this),y.Zvb(Q,T.Aib(o),h)),h.d=a.fJ(u,h.d),h.d.length<=h.c.length&&(h.a=h.d.length));for(D=0;D<A.f;D++)P[D]==0&&(o=A.C[D]+(A.A[D]<<16),h=y.opb(Q,T.Aib(o)),h||(h=new a.pJ(this),y.Zvb(Q,T.Aib(o),h)),h.c=a.fJ(D,h.c),h.c.length<=h.d.length&&(h.a=h.c.length));for(this.a=new y.Mj(),b=(s=new y.Oob(Q).a.bd().Tb(),new y.Vob(s));b.a.dd();)h=(L=b.a.ed(),L.hd()),a.nJ(h)||y.sj(this.a,h);},N(399,1,{},a.dJ),U.b=0,U.g=0,l.L9=O(399),a.eJ=function(t){t.d=j(l.j7,H,6,0,15,1),t.c=j(l.j7,H,6,0,15,1);},a.fJ=function(t,A){var n,P;for(P=j(l.j7,H,6,A.length+1,15,1),n=0;n<A.length;n++)P[n]=A[n];return P[A.length]=t,P;},a.gJ=function(t,A){var n,P,r,i,D,u;for(P=t,r=0,i=P.length;r<i;++r)if(n=P[r],A.j[n]!=0)return  false;for(D=e.Bl(A,t[0]),u=1;u<t.length;u++)if(e.Bl(A,t[u])!=D)return  false;return  true;},a.hJ=function(t,A,n){var P,r,i,D,u,o;if(o=y.xj(t.b,A),t.d.length<=t.c.length)for(u=0;u<t.a;u++)n[t.d[u]]=t.c[o[u]];else {for(r=t.d,i=0,D=r.length;i<D;++i)P=r[i],n[P]=-1;for(u=0;u<t.a;u++)n[t.d[o[u]]]=t.c[u];}},a.iJ=function(t){var A,n,P;for(P=d.Math.max(t.d.length,t.c.length),n=1,A=P-t.a+1;A<=P;A++)n*=A;return n;},a.jJ=function(t,A,n,P){var r,i;if(t.d.length>t.c.length)for(r=0;r<t.a;r++)n[P+r]=t.c[r];else for(i=y.xj(t.b,A),r=0;r<t.a;r++)n[P+r]=t.c[i[r]];},a.kJ=function(t,A,n,P){var r,i;if(t.d.length<=t.c.length)for(r=0;r<t.a;r++)n[P+r]=t.d[r];else for(i=y.xj(t.b,A),r=0;r<t.a;r++)n[P+r]=t.d[i[r]];},a.lJ=function(t){var A,n;return t.b=new y.Mj(),n=j(l.j7,H,6,t.a,15,1),A=j(l.Feb,ht,6,d.Math.max(t.d.length,t.c.length),16,1),a.oJ(t,0,A,n),t.b.a.length;},a.mJ=function(t,A,n){var P,r;if(r=y.xj(t.b,A),t.d.length<=t.c.length){for(P=0;P<t.a;P++)if(n[t.d[P]]!=null&&n[t.d[P]][t.c[r[P]]])return  true;}else for(P=0;P<t.a;P++)if(n[t.d[r[P]]]!=null&&n[t.d[r[P]]][t.c[P]])return  true;return  false;},a.nJ=function(t){var A,n,P;if(t.a==0)return  true;if(t.d.length==1&&t.c.length==1)return (t.e.i==null||t.e.i[t.d[0]]==null||!t.e.i[t.d[0]][t.c[0]])&&(++t.e.b,t.e.f[t.d[0]]=t.e.b,t.e.d[t.c[0]]=t.e.b),true;if(a.gJ(t.d,t.e.e)||a.gJ(t.c,t.e.c)){for(A=d.Math.min(t.d.length,t.c.length),P=0;A!=0&&P<t.d.length;P++)for(n=0;n<t.c.length;n++)if(t.e.d[t.c[n]]==0&&(t.e.i==null||t.e.i[t.d[P]]==null||!t.e.i[t.d[P]][t.c[n]])){--A,++t.e.b,t.e.f[t.d[P]]=t.e.b,t.e.d[t.c[n]]=t.e.b;break;}return  true;}return  false;},a.oJ=function(t,A,n,P){var r;for(r=0;r<n.length;r++)n[r]||(n[r]=true,P[A]=r,A+1==P.length?y.sj(t.b,y.Jqb(P,P.length)):a.oJ(t,A+1,n,P),n[r]=false);},a.pJ=function(t){this.e=t,a.eJ(this);},N(201,1,{201:1},a.pJ),U.a=0,l.K9=O(201),a.qJ=function(t,A){return t.b<A.b?-1:t.b>A.b?1:t.a<A.a?-1:t.a>A.a?1:0;},a.rJ=function(t,A){this.b=t,this.a=A;},N(125,1,{125:1,32:1},a.rJ),U.Sb=function(t){return a.qJ(this,t);},U.a=0,U.b=0,l.P9=O(125),a.tJ=function(t,A){var n;return n=a.xJ(t.a,A),t.a=a.wJ(t.a,n),n;},a.uJ=function(t){t.a=t.b;},a.vJ=function(){this.b=new a.zJ(),this.a=this.b;},N(398,1,{},a.vJ),l.M9=O(398),a.wJ=function(t,A){return t.a[A];},a.xJ=function(t,A){if(t.b==-1)t.a=j(l.N9,C1,170,A,0,1),t.b=A,t.c=A-1;else do t.c=t.c==0?t.b-1:t.c-1;while(t.a[t.c]&&a.yJ(t.a[t.c]));return !t.a[t.c]&&(t.a[t.c]=new a.zJ()),t.c;},a.yJ=function(t){var A,n,P,r,i;if(!t.d)if(t.b==-1)t.d=true;else {for(i=false,n=t.a,P=0,r=n.length;P<r;++P)if(A=n[P],!A)i=true;else if(!a.yJ(A))return  false;t.d=!i;}return t.d;},a.zJ=function(){this.b=-1;},N(170,1,{170:1},a.zJ),U.b=0,U.c=0,U.d=false,l.N9=O(170),a.AJ=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h,b;for(b=j(l.Feb,ht,6,P.length,16,1),u=t,Q=0,s=u.length;Q<s;++Q)if(i=u[Q],A[i]==0){if(P[n[i]]||b[n[i]])return  false;b[n[i]]=true;}for(h=-1,D=t,o=0,L=D.length;o<L;++o)if(i=D[o],A[i]==0){if(h==-1)h=r[i];else if(r[i]!=h)return  false;}return  true;},a.BJ=function(t){var A,n,P,r,i,D,u,o;for(t.t=new y.Mj(),P=1,i=0,n=0;n<t.u.f;n++)e.Ll(t.u,n)!=0&&!e.Am(t.u,n)&&(i=d.Math.max(i,e.Ll(t.u,n)));if(i!=0){for(r=j(l.j7,H,6,i+1,15,1),A=0;A<t.u.f;A++)e.Ll(t.u,A)!=0&&!e.Am(t.u,A)&&(r[e.Ll(t.u,A)]=A+1);for(u=0;u<t.D.f;u++)o=e.Ll(t.D,u),o!=0&&o<=i&&!e.Am(t.D,u)&&r[o]!=0&&(D=r[o]-1,t.I[u]=P,t.B[D]=P++,y.sj(t.t,new a.rJ(u,D)));}return t.t.a.length;},a.CJ=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b;for(s=a.FJ(t.D),u=a.FJ(t.u),D=j(l.sdb,i1,44,8,0,1),b=a.DJ(t.D,s),Q=a.DJ(t.u,u),L=0;L<=7;L++)for(D[L]=new y.gwb(new Lt.K3()),i=(P=new y.pwb(new y.vwb(new y.Cpb(b[L]).a).b),new y.Lpb(P));y.kob(i.a.a);)r=(n=y.nwb(i.a),n.gd()),h=y.opb(b[L],r),o=y.opb(Q[L],r),o!=null&&(A=j(l.j7,E1,7,2,0,2),A[0]=h,A[1]=o,y.Zvb(D[L],r,A));return D;},a.DJ=function(t,A){var n,P,r,i,D;for(i=j(l.sdb,i1,44,8,0,1),D=0;D<=7;D++)for(i[D]=new y.gwb(new Lt.K3()),n=0;n<t.f;n++)r=A[n][D],r!=null&&(P=y.opb(i[D],r),P=P==null?j(l.j7,H,6,1,15,1):y.Jqb(P,P.length+1),P[P.length-1]=n,y.Zvb(i[D],r,P));return i;},a.EJ=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;for(D=j(l.j7,H,6,t.s,15,1),r=0;r<A.f;r++){for(b=A.j[r]+(A.i[r].length-A.e[r]),h=0,o=0;o<A.i[r].length;o++)if(o<A.j[r]||o>=A.e[r]){for(c=2*n[A.i[r][o]],i=A.k[r][o],e.fm(A,i)==2&&(i<A.g&&e.pt(A.p,i)||++c),L=0;L<h&&!(c<D[L]);L++);for(s=h;s>L;s--)D[s]=D[s-1];D[L]=c,++h;}for(e.eh(t.b[r],r),e.ah(t.b[r],t.a,l.Ueb(n[r])),Q=b;Q<t.s;Q++)e.ah(t.b[r],t.a+1,0);for(u=0;u<b;u++)e.ah(t.b[r],t.a+1,l.Ueb(D[u]));}for(P=A.f;P<t.b.length;P++)for(e.eh(t.b[P],P),e.ah(t.b[P],t.a,Mt(A.f+1)),u=0;u<t.s;u++)e.ah(t.b[P],t.a+1,0);},a.FJ=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(e.bw(t,7),o=new e.uw(t.f,t.g),u=y1(l.f7,[bP,bA],[58,10],0,[t.f,8],2),P=j(l.j7,H,6,t.f,15,1),r=j(l.Feb,ht,6,t.f,16,1),i=j(l.Feb,ht,6,t.g,16,1),g=0;g<t.f;g++){for(g!=0&&(y.Xqb(r,r.length),y.Xqb(i,i.length)),u[g][0]=j(l.f7,Y,6,2,15,1),u[g][0][0]=t.C[g]<<24>>24,u[g][0][1]=t.A[g]<<24>>24,P[0]=g,r[g]=true,b=0,h=1,t.u[g]|=512,B=1;B<=7&&h<t.f;B++){for(c=h,Q=b;Q<h;Q++)for(n=P[Q],L=0;L<t.j[n];L++)D=t.i[n][L],r[D]||(r[D]=true,P[c++]=D),i[t.k[n][L]]=true;if(c==h)break;for(b=h,h=c,e.Do(t,o,i,true,null),A=0;A<o.q;A++)o.s[A]=0,o.T=0,e.wn(o,A,0);u[g][B]=w.Czb((s=e.rg(new e.Og(o,16)),S1.Mmb(),s));}t.u[g]&=-513;}return u;},a.GJ=function(t,A){var n,P,r;for(n=0,y.brb(t.b),P=0;P<A.length;P++)(P==0||e.dh(t.b[P],t.b[P-1])!=0)&&++n,r=t.b[P].a,A[r]=n;return n;},a.HJ=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h;for(h=new y.Ywb(new Lt._3()),Q=A,L=0,s=Q.length;L<s;++L)for(o=Q[L],r=n,i=0,D=r.length;i<D;++i)P=r[i],u=j(l.j7,H,6,2,15,1),u[0]=t.J[o],u[1]=t.C[P],y.Zvb(h.a,u,(T.Qgb(),Fn))==null;return y.Mi(h,j(l.j7,E1,7,0,0,2));},a.IJ=function(t,A,n){return e.$o(t.u,n)!=0&&(t.H[t.G[A]]=true),e.$o(t.D,A)!=0&&(t.A[t.w[n]]=true),new a.rJ(A,n);},a.JJ=function(t,A,n,P,r){var i,D,u,o,Q;for(i=n[P],Q=false,u=0;u<A.f;u++)if(u!=P&&n[u]==i){Q=true;break;}if(!Q)return r;for(D=0;D<A.f;D++)(D==P||n[D]>i)&&++n[D];++r;do o=r,a.EJ(t,A,n),r=a.GJ(t,n);while(o!=r);return r;},a.KJ=function(t){var A,n,P,r,i,D,u;for(D=j(l.f7,bP,58,8,0,3),u=0;u<=7;u++)for(D[u]=j(l.f7,bA,10,t[u].c,0,2),P=0,i=(n=new y.pwb(new y.vwb(new y.Cpb(t[u]).a).b),new y.Lpb(n));y.kob(i.a.a);)r=(A=y.nwb(i.a),A.gd()),D[u][P++]=r;return D;},a.LJ=function(t,A){var n,P,r,i,D,u,o,Q,L;for(L=0,r=t,D=0,o=r.length;D<o;++D)n=r[D],A[n]==0&&++L;if(L==0)return null;for(Q=j(l.j7,H,6,L,15,1),L=0,P=t,i=0,u=P.length;i<u;++i)n=P[i],A[n]==0&&(Q[L++]=n);return Q;},a.MJ=function(t){return t.K++==64?false:t.i?a.yJ(t.i.b)?false:(a.SJ(t),true):(t.i=new a.vJ(),true);},a.NJ=function(t){var A,n,P,r,i;for(i=d.Math.max(t.D.f,t.u.f),t.a=e.Pg(i),t.s=2,n=0;n<t.D.f;n++)t.s=d.Math.max(t.s,e.$o(t.D,n)+e.rp(t.D,n));for(P=0;P<t.u.f;P++)t.s=d.Math.max(t.s,e.$o(t.u,P)+e.rp(t.u,P));for(r=d.Math.max(2,(62+t.a+t.s*(t.a+5))/63|0),t.b=j(l.v7,He,101,i,0,1),A=0;A<i;A++)t.b[A]=new e.fh(r);t.F=new e.Og(t.D,1),t.v=new e.Og(t.u,1);},a.OJ=function(t){var A,n,P,r,i,D,u,o,Q;for(Q=t.F.g,t.J=y.Jqb(Q,Q.length),A=t.v.g,t.C=y.Jqb(A,A.length),r=t.J,D=0,o=r.length;D<o;++D)n=r[D],t.o<n&&(t.o=n);for(P=t.C,i=0,u=P.length;i<u;++i)n=P[i],t.n<n&&(t.n=n);},a.PJ=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g;if(g=a.LJ(A,t.I),g==null||(c=a.LJ(n,t.B),c==null))return null;if(g.length==1&&c.length==1)return new a.rJ(g[0],c[0]);if(b=a.HJ(t,g,c),b.length==1)return new a.rJ(g[0],c[0]);for(L=a.tJ(t.i,b.length),h=-1,i=g,u=0,Q=i.length;u<Q;++u)if(P=i[u],t.J[P]==b[L][0]){h=P;break;}for(s=-1,r=c,D=0,o=r.length;D<o;++D)if(P=r[D],t.C[P]==b[L][1]){s=P;break;}return t.o=a.JJ(t,t.D,t.J,h,t.o),t.n=a.JJ(t,t.u,t.C,s,t.n),new a.rJ(h,s);},a.QJ=function(t){var A,n,P,r,i;for(P=a.RJ(t);P;){for(i=false,n=0;n<e.$o(t.D,P.b);n++)t.I[e.Zo(t.D,P.b,n)]==0&&(i=true);for(r=false,A=0;A<e.$o(t.u,P.a);A++)t.B[e.Zo(t.u,P.a,A)]==0&&(r=true);if(i&&r)break;t.I[P.b]=cP,t.B[P.a]=cP,P=a.RJ(t);}if(!P){for(n=0;n<t.I.length;n++)t.I[n]==cP&&(t.I[n]=0);for(A=0;A<t.B.length;A++)t.B[A]==cP&&(t.B[A]=0);}return P;},a.RJ=function(t){for(var A,n,P,r,i;t.t.a.length!=0;)if(P=y.Ej(t.t,0),t.I[P.b]==0&&t.B[P.a]==0)return P;for(;t.g>=0;){for(;t.g>=2&&t.c<t.j[t.g].length;){if(n=t.j[t.g][t.c],A=y.opb(t.k[t.g],n),e.Tl(t.D,A[0][0])==6&&(P=a.PJ(t,A[0],A[1]),P))return P;++t.c;}for(;t.g>=2&&t.d<t.j[t.g].length;){if(n=t.j[t.g][t.d],A=y.opb(t.k[t.g],n),e.Tl(t.D,A[0][0])!=6&&(P=a.PJ(t,A[0],A[1]),P))return P;++t.d;}for(;t.p&&t.g==0&&t.e<t.j[0].length;)if(n=t.j[0][t.e++],A=y.opb(t.k[t.g],n),A[0].length==1&&A[1].length==1&&(P=a.TJ(t,A[0][0],A[1][0]),P))return P;for(;t.g==0&&t.f<t.j[t.g].length;)if(n=t.j[t.g][t.f++],A=y.opb(t.k[t.g],n),A[0].length>=A[1].length&&a.AJ(A[0],t.I,t.G,t.H,t.J)||A[1].length>=A[0].length&&a.AJ(A[1],t.B,t.w,t.A,t.C)){for(i=0,r=0;i<A[0].length&&t.I[A[0][i]]!=0;)++i;for(;r<A[1].length&&t.B[A[1][r]]!=0;)++r;if(i<A[0].length&&r<A[1].length&&(P=a.TJ(t,A[0][i],A[1][r]),P))return P;}--t.g,t.c=0,t.d=0,t.e=0,t.f=0;}return null;},a.SJ=function(t){y.Nqb(t.I,0),y.Nqb(t.B,0),y.Rqb(t.H),y.Rqb(t.A),a.OJ(t),a.uJ(t.i),t.q=a.BJ(t),t.g=7,t.c=0,t.d=0,t.e=0,t.f=0;},a.TJ=function(t,A,n){return t.L!=null&&t.L[A]!=null&&t.L[A][n]?null:t.I[A]==0&&t.B[n]==0?a.IJ(t,A,n):null;},a.UJ=function(t,A,n,P,r){var i,D,u,o,Q;for(this.D=t,this.u=A,this.I=n,this.B=P,this.L=r,a.NJ(this),a.OJ(this),this.k=a.CJ(this),this.j=a.KJ(this.k),this.r=0,Q=(u=new y.pwb(new y.vwb(new y.Cpb(this.k[0]).a).b),new y.Lpb(u));y.kob(Q.a.a);)o=(D=y.nwb(Q.a),D.gd()),i=y.opb(this.k[0],o),this.r+=d.Math.min(i[0].length,i[1].length);this.p=this.r==this.D.f&&this.D.f==this.u.f,this.G=j(l.j7,H,6,this.D.f,15,1),this.w=j(l.j7,H,6,this.u.f,15,1),this.H=j(l.Feb,ht,6,e.hp(this.D,this.G,false,false),16,1),this.A=j(l.Feb,ht,6,e.hp(this.u,this.w,false,false),16,1),this.q=a.BJ(this),this.g=7,this.K=0;},N(397,1,{},a.UJ),U.a=0,U.c=0,U.d=0,U.e=0,U.f=0,U.g=0,U.n=0,U.o=0,U.p=false,U.q=0,U.r=0,U.s=0,U.K=0,l.O9=O(397),a.VJ=function(t,A,n){var P,r,i,D,u;for(this.a=j(l.j7,H,6,t.f,15,1),this.a[0]=A,n[A]=true,i=0,D=0;i<=D;){for(r=e.$o(t,this.a[i]),u=0;u<r;u++)P=e.Zo(t,this.a[i],u),d.Math.abs(t.w[P])==0&&H1(J(t.B[P],r0),0)&&!n[P]&&(this.a[++D]=P,n[P]=true);++i;}},N(395,1,{},a.VJ),l.Q9=O(395),a.WJ=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q;for(e.bw(t,7),L=new e.uw(t.f,t.g),Q=y1(l.f7,[c0,bP],[204,58],0,[t.f,8],2),P=j(l.j7,H,6,t.f,15,1),r=j(l.j7,H,6,t.f,15,1),i=j(l.Feb,ht,6,t.f,16,1),p=0;p<t.f;p++)if(Q[p]=y1(l.f7,[bP,bA],[58,10],0,[t.j[p],8],2),!(A&&t.C[p]!=6))for(o=0;o<t.j[p];o++){for(s=t.i[p][o],y.Xqb(i,i.length),k=1,B=2,t.u[s]|=512,q=0;q<8&&B<t.f;q++){if(q==0)P[0]=s,i[s]=true,P[1]=p,i[p]=true;else {for(M=B,b=k;b<B;b++)for(n=P[b],c=0;c<t.j[n];c++)u=t.i[n][c],!i[u]&&(!A||t.C[u]==6)&&(i[u]=true,P[M++]=u);if(M==B){if(!A)for(h=q;h<8;h++)Q[p][o][h]=Q[p][o][h-1];break;}k=B,B=M;}if(q==0)Q[p][o][q]=j(l.f7,Y,6,2,15,1),Q[p][o][q][0]=t.C[p]<<24>>24,Q[p][o][q][1]=t.A[p]<<24>>24;else {for(e.Co(t,L,i,true,r),e.jn(L,r[s],0),e.wn(L,r[s],0),n=0;n<t.f;n++)r[n]!=-1&&t.j[n]>e.$o(L,r[n])&&e.vn(L,r[n],Z1,true);if(A)for(D=0;D<L.g;D++)L.H[D]=1,L.T=0;Q[p][o][q]=w.Czb((g=e.rg(new e.Og(L,16)),S1.Mmb(),g));}}t.u[s]&=-513;}return Q;},a.XJ=function(t,A,n){var P,r;for(P=0,r=0;r<t.j[A];r++)e.Tl(t,t.i[A][r])==n&&++P;return P;},a.YJ=function(t,A,n,P,r){var i,D,u,o,Q,L,s,h;for(h=0,s=-1,D=0;D<t.j.f;D++){L=y.xj(A.g,h),e.bw(L,1);do++s==L.f&&(s=0,L=At.pH(A,++h));while(L.K&&pt(J(L.B[s],r0),0));e.rn(L,s,n[D],n[D]<=r);}for(Q=0,o=-1,i=0;i<t.d.f;i++){u=y.xj(A.f,Q),e.bw(u,1);do++o==u.f&&(o=0,u=At.oH(A,++Q));while(u.K&&pt(J(u.B[o],r0),0));e.rn(u,o,P[i],P[i]<=r);}},a.ZJ=function(t,A,n,P,r){var i,D,u,o,Q;for(Q=-1,D=0;D<e.$o(t.j,n);D++)if(e.Zo(t.j,n,D)==A){Q=D;break;}for(u=-1,i=0;i<e.$o(t.d,r);i++)if(e.Zo(t.d,r,i)==P){u=i;break;}for(o=0;o<8;o++)if(Lt.J3(t.k[n][Q][o],t.e[r][u][o])!=0)return o<<8;return j1;},a.$J=function(t,A){return A<t.g&&(e.ut(t.p,A)||t.H[A]==8)?0:t.H[A]&127;},a._J=function(t,A,n){var P,r;for(r=0;r<t.j[n];r++)if(P=t.i[n][r],P!=A)return P;return  -1;},a.aK=function(t,A){var n,P;for(n=t.o[A],P=0;P<t.j[A];P++)e.Hm(t,t.i[A][P])&&++n;return n;},a.bK=function(t,A,n,P,r){var i,D,u,o,Q;for(Q=-1,D=0;D<e.$o(t.j,n);D++)if(e.Zo(t.j,n,D)==A){Q=D;break;}for(u=-1,i=0;i<e.$o(t.d,r);i++)if(e.Zo(t.d,r,i)==P){u=i;break;}for(o=2;o<8;o++)if(t.p[n][Q][o]==null||Lt.J3(t.p[n][Q][o],t.i[r][u][o])!=0)return o-2<<8;return 1536;},a.cK=function(t){var A,n,P,r,i,D,u,o;for(u=j(l.j7,H,6,t.f,15,1),o=(e.bw(t,7),t.p),i=0;i<d.Math.min(32,o.j.a.length);i++)for(D=y.xj(o.j,i),n=D,P=0,r=n.length;P<r;++P)A=n[P],u[A]|=1<<i;return u;},a.dK=function(t,A,n,P){return (P[t]&P[A]&~P[n])!=0;},a.eK=function(t,A,n,P,r,i){var D,u,o,Q,L,s,h,b;for(t.j=A,t.d=n,t.n=j(l.j7,H,6,P.length,15,1),t.f=j(l.j7,H,6,r.length,15,1),t.k=a.WJ(t.j,false),t.p=a.WJ(t.j,true),t.e=a.WJ(t.d,false),t.i=a.WJ(t.d,true),t.o=a.cK(t.j),t.g=a.cK(t.d),t.q=xr,s=new a.UJ(A,n,t.n,t.f,i),t.a=0;a.MJ(s);){for(++t.a,t.c=s.q,Q=s.r,L=a.QJ(s);L;)a.fK(t,L),L=a.QJ(s);if(t.b=t.c,t.c<Q?(D=new a.dJ(t.j,t.d,t.n,t.f,t.c,i),h=a.cJ(D),t.c+=D.b-D.g):(b=new a.bJ(t.j,t.d),h=a.aJ(b,a.VI(b,t.n,t.f))),t.q<h){for(t.q=h,o=0;o<P.length;o++)P[o]=t.n[o];for(u=0;u<r.length;u++)r[u]=t.f[u];}}},a.fK=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z;for(Q=j(l.j7,H,6,t.j.f,15,1),k=j(l.j7,H,6,t.d.f,15,1),L=j(l.j7,H,6,t.j.f,15,1),p=j(l.j7,H,6,t.d.f,15,1),Q[0]=A.b,L[A.b]=A.b,k[0]=A.a,p[A.a]=A.a,t.n[A.b]==0&&(++t.c,t.n[A.b]=t.c,t.f[A.a]=t.c),u=0,s=0;u<=s;){for(V=Q[u],q=k[u],B=y1(l.j7,[E1,H],[7,6],15,[e.$o(t.j,V),e.$o(t.d,q)],2),b=0;b<e.$o(t.j,V);b++)if(S=e.Zo(t.j,V,b),t.n[S]==0)for(i=a.$J(t.j,e._o(t.j,V,b)),g=0;g<e.$o(t.d,q);g++)M=e.Zo(t.d,q,g),t.f[M]==0&&e.Tl(t.j,S)==e.Tl(t.d,M)&&(D=e._o(t.d,q,g),Z=a.bK(t,V,S,q,M),i==a.$J(t.d,D)||Z!=0?a.iK(t,V,S,q,M)&&(o=a.ZJ(t,V,S,q,M),I=d.Math.max(Z,o),I!=0&&(c=a.gK(t,L[V],V,S,p[q],q,M),a.jK(t,V,S,q,M,I,c)&&(B[b][g]=I,c&&(B[b][g]+=64),a.dK(L[V],V,S,t.o)==a.dK(p[q],q,M,t.g)&&(B[b][g]+=128),a.aK(t.j,S)==a.aK(t.d,M)&&(B[b][g]+=1)))):a.ZJ(t,V,S,q,M)==j1&&(B[b][g]=768));for(;;){for(n=0,r=-1,P=-1,h=0;h<B.length;h++)if(S=e.Zo(t.j,V,h),t.n[S]==0)for(g=0;g<B[h].length;g++)M=e.Zo(t.d,q,g),t.f[M]==0&&n<B[h][g]&&(n=B[h][g],r=S,P=M);if(n==0)break;++s,Q[s]=r,L[r]=Q[u],k[s]=P,p[P]=k[u],++t.c,t.n[r]=t.c,t.f[P]=t.c;}++u;}return s+1;},a.gK=function(t,A,n,P,r,i,D){var u,o,Q,L,s;if(e.$o(t.j,n)==3&&(e.Nl(t.j,n)==1||e.Nl(t.j,n)==2)&&e.$o(t.d,i)==3&&(e.Nl(t.d,i)==1||e.Nl(t.d,i)==2)){for(s=A>P,Q=-1,o=0;o<e.$o(t.j,n);o++)if(Q=e.Zo(t.j,n,o),Q!=A&&Q!=P){(Q>P&&Q<A||Q<P&&Q>A)&&(s=!s);break;}for(L=r>D,u=0;u<e.$o(t.d,i);u++)if(Q=e.Zo(t.d,i,u),Q!=r&&Q!=D){(Q>D&&Q<r||Q<D&&Q>r)&&(L=!L);break;}return s==L==(e.Nl(t.j,n)==e.Nl(t.d,i));}return  true;},a.hK=function(t,A){var n,P,r,i;for(t.j=new e.tw(),P=0;P<A.g.a.length;P++)i=y.xj(A.g,P),e.bw(i,1),e.Zk(t.j,i,i.f,i.g,!i.K);for(t.d=new e.tw(),n=0;n<A.f.a.length;n++)r=y.xj(A.f,n),e.bw(r,1),e.Zk(t.d,r,r.f,r.g,!r.K);e.lq(t.j),e.lq(t.d);},a.iK=function(t,A,n,P,r){var i,D;return !(e.Tl(t.j,n)==6&&d.Math.abs(e.$o(t.j,n)-e.$o(t.d,r))>1||e.Tl(t.j,n)==8&&e.$o(t.j,n)==2&&a.XJ(t.j,A,8)>a.XJ(t.d,P,8)||e.Tl(t.d,r)==8&&e.$o(t.d,r)==2&&a.XJ(t.j,A,8)<a.XJ(t.d,P,8)||e.Tl(t.j,n)==8&&e.$o(t.j,n)==2&&e.$o(t.d,r)==2&&(D=e.Zo(t.j,n,0)==A?0:1,i=e.Zo(t.d,r,0)==P?0:1,Lt.J3(t.p[n][D][3],t.i[r][i][3])!=0)||e.Tl(t.j,n)==7&&a.XJ(t.j,A,7)<a.XJ(t.d,P,7)||e.Tl(t.d,r)==7&&a.XJ(t.j,A,7)>a.XJ(t.d,P,7)||e.Yo(t.j,e.Xo(t.j,A,n))==3^e.Yo(t.d,e.Xo(t.d,P,r))==3);},a.jK=function(t,A,n,P,r,i,D){return !(e.Tl(t.j,n)==8&&e.Qo(t.j,n)==0&&a.nK(t.j,A)&&a.nK(t.d,P)&&i!=j1||e.Tl(t.j,n)==8&&e.$o(t.j,n)==2&&a.nK(t.j,a._J(t.j,A,n))&&i<768||e.Tl(t.j,A)==8&&(a.nK(t.j,n)||a.nK(t.d,r))&&i<512||!D&&(e.Tl(t.j,n)!=6||!a.lK(t.j,A))||e.Tl(t.j,n)==8&&e.Tl(t.j,A)!=6&&i!=j1||e.Tl(t.j,A)==5&&e.Tl(t.j,n)==6&&i<768);},a.kK=function(){},a.lK=function(t,A){var n;for(n=0;n<t.j[A];n++)if(e.Tl(t,t.i[A][n])!=6)return  true;return  false;},a.mK=function(t,A,n){var P;for(P=0;P<t.j[A];P++)if(e.Tl(t,t.i[A][P])!=6&&t.i[A][P]!=n)return  true;return  false;},a.nK=function(t,A){var n;for(n=0;n<t.j[A];n++)if(t.n[A][n]==2&&e.Tl(t,t.i[A][n])>6)return  true;return  false;},a.oK=function(t,A,n){var P;for(P=0;P<t.j[A];P++)if(t.n[A][P]==2&&e.Tl(t,t.i[A][P])>6&&t.i[A][P]!=n)return  true;return  false;},N(394,1,{},a.kK),U.a=0,U.b=0,U.c=0,U.q=0,l.R9=O(394),E.sK=function(){E.sK=D1,E.qK=v(F(l.Ybb,1),st,2,6,["","r0","r","r3","r3-4","r3-5","r3-6","r3-7","r4","r4-5","r4-6","r4-7","r>3","r5","r5-6","r5-7","r>4","r6","r6-7","r>5","r7","r>6","r>7"]),E.pK=v(F(l.Ybb,1),st,2,6,[wA,BD,"is in any ring","3 members","3-4 members","3-5 members","3-6 members","3-7 members","4 members","4-5 members","4-6 members","4-7 members","> 3 members","5 members","5-6 members","5-7 members","> 4 members","6 members","6-7 members","> 5 members","7 members","> 6 members","> 7 members"]),E.rK=v(F(l.k7,1),e0,6,14,[0,QA,Z8,SA,25769803776,60129542144,128849018880,266287972352,VA,51539607552,120259084288,257698037760,532575944704,xA,103079215104,240518168576,515396075520,zA,206158430208,481036337152,NA,412316860416,De]);},E.tK=function(t,A,n,P){var r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S;for(t.A=A,t.a=n,s=Rt(d.Math.round((d1.NN==-1&&(d1.NN=d1.PN()),d1.NN*8))),h=v(F(l.j7,1),H,6,15,[s,s/2|0,s*3/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s*3/2|0,s/4|0,s/2|0,s/2|0,s/4|0]),b=v(F(l.j7,1),H,6,15,[s*3/2|0,s/2|0]),c=v(F(l.j7,1),H,6,15,[s,-2,s,-2,s]),S=j(l.j7,H,6,1+2*h.length+(P?2*b.length:0),15,1),B=0,D=h,o=0,L=D.length;o<L;++o)r=D[o],S[B++]=r,S[B++]=-2;if(P)for(i=b,u=0,Q=i.length;u<Q;++u)r=i[u],S[B++]=r,S[B++]=-2;for(S[B++]=s,C.z2(t.v,c,S),t.b=C.t2(t.v,"Any atomic number"),C.V1(t.b,t),C.s2(t.v,t.b,1,1,3,1),t.w=C.v2(t.v,zr),t.D=C.w2(t.v,16,1),C.r2(t.v,t.w,1,3),C.r2(t.v,t.D,3,3),t.g=C.u2(t.v),C.j2(t.g,wA),C.j2(t.g,"is aromatic"),C.j2(t.g,"is hetero-aromatic"),C.j2(t.g,"is not aromatic"),C.r2(t.v,C.v2(t.v,"Aromaticity:"),1,5),C.r2(t.v,t.g,3,5),t.s=C.u2(t.v),C.j2(t.s,wA),C.j2(t.s,"0 (not in a ring)"),C.j2(t.s,"0 or 2 (0 or 1 ring)"),C.j2(t.s,">=2 (any ring count)"),C.j2(t.s,"2 (in 1 ring)"),C.j2(t.s,"3 (bridge head; 2 rings)"),C.j2(t.s,">3 (in more than 2 rings)"),C.r2(t.v,C.v2(t.v,"Ring bonds:"),1,7),C.r2(t.v,t.s,3,7),t.t=C.u2(t.v),C.j2(t.t,wA),C.j2(t.t,"3 members"),C.j2(t.t,"4 members"),C.j2(t.t,"5 members"),C.j2(t.t,"6 members"),C.j2(t.t,"7 members"),C.r2(t.v,C.v2(t.v,"Smallest ring size:"),1,9),C.r2(t.v,t.t,3,9),t.r=C.u2(t.v),M=E.pK,p=0,q=M.length;p<q;++p)k=M[p],C.j2(t.r,k);for(C.r2(t.v,C.v2(t.v,"Any ring size:"),1,11),C.r2(t.v,t.r,3,11),t.i=C.u2(t.v),C.j2(t.i,wA),C.j2(t.i,"not charged"),C.j2(t.i,"has negative charge"),C.j2(t.i,"has positive charge"),C.r2(t.v,C.v2(t.v,"Charge:"),1,13),C.r2(t.v,t.i,3,13),t.n=C.u2(t.v),C.j2(t.n,wA),C.j2(t.n,wP),C.j2(t.n,gP),C.j2(t.n,"exactly 3"),C.j2(t.n,Nr),C.j2(t.n,dD),C.j2(t.n,jP),C.j2(t.n,Ir),C.j2(t.n,Or),C.j2(t.n,mD),C.j2(t.n,Gr),C.j2(t.n,"1,2, or 3"),C.j2(t.n,"2 or 3"),C.r2(t.v,C.v2(t.v,"Non-H neighbours:"),1,15),C.r2(t.v,t.n,3,15),t.j=C.u2(t.v),C.j2(t.j,wA),C.j2(t.j,"exactly 0"),C.j2(t.j,wP),C.j2(t.j,gP),C.j2(t.j,"exactly 3"),C.j2(t.j,kD),C.j2(t.j,Nr),C.j2(t.j,dD),C.j2(t.j,jP),C.j2(t.j,Ir),C.j2(t.j,Or),C.j2(t.j,mD),C.j2(t.j,Gr),C.j2(t.j,"1,2, or 3"),C.j2(t.j,"2 or 3"),C.r2(t.v,C.v2(t.v,"Electroneg. neighbours:"),1,17),C.r2(t.v,t.j,3,17),t.k=C.u2(t.v),C.j2(t.k,wA),C.j2(t.k,"none"),C.j2(t.k,wP),C.j2(t.k,gP),C.j2(t.k,jP),C.j2(t.k,Ir),C.j2(t.k,Or),C.j2(t.k,kD),C.j2(t.k,Nr),C.j2(t.k,Gr),C.r2(t.v,C.v2(t.v,"Hydrogen count:"),1,19),C.r2(t.v,t.k,3,19),t.p=C.u2(t.v),C.j2(t.p,wA),C.j2(t.p,"none"),C.j2(t.p,wP),C.j2(t.p,gP),C.j2(t.p,jP),C.r2(t.v,C.v2(t.v,"Pi-electron count:"),1,21),C.r2(t.v,t.p,3,21),t.o=C.u2(t.v),C.j2(t.o,wA),g=1;g<15;g++)C.j2(t.o,""+(g-7));C.r2(t.v,C.v2(t.v,"Oxidation state:"),1,23),C.r2(t.v,t.o,3,23),t.c=C.t2(t.v,"prohibit further substitution"),C.V1(t.c,t),C.s2(t.v,t.c,1,25,3,25),t.f=C.t2(t.v,"require further substitution"),C.V1(t.f,t),C.s2(t.v,t.f,1,27,3,27),t.u=C.u2(t.v),C.j2(t.u,wA),C.j2(t.u,"is a stereo center"),C.j2(t.u,"is not a stereo center"),C.r2(t.v,C.v2(t.v,"Stereo center:"),1,29),C.r2(t.v,t.u,3,29),t.e=C.t2(t.v,"match stereo center"),C.s2(t.v,t.e,1,31,3,31),t.d=C.t2(t.v,"is part of exclude group"),C.s2(t.v,t.d,1,33,3,33),P&&(C.s2(t.v,C.v2(t.v,"Stereo center hint for product:"),1,35,3,35),t.q=C.u2(t.v),C.j2(t.q,"Copy from generic product"),C.j2(t.q,"Keep reactant configuration"),C.j2(t.q,"Invert reactant configuration"),C.j2(t.q,"Racemise configuration"),C.s2(t.v,t.q,1,37,3,37)),e.bw(t.A,31),E.wK(t);},E.uK=function(t){var A,n,P,r,i,D,u,o,Q;for(u=null,o=T.Cjb(C.r3(t.D));w.aAb(o).length!=0;){if(n=T.rjb(o,Y1(44)),n==-1?(D=T.Cjb(o),o=""):(D=T.Cjb((w.Wzb(0,n,w.aAb(o).length),w.aAb(o).substr(0,n))),o=T.Cjb((w.Xzb(n+1,w.aAb(o).length+1),w.aAb(o).substr(n+1)))),T.kjb(D,"X")){w.aAb(o).length!=0&&(o=","),o=(w.Qzb(o),o+(w.Qzb("F,Cl,Br,I"),"F,Cl,Br,I"));continue;}if(A=(e.Rk(),e.po(D,321)),A!=0)if(A==1)C.B2(t.v,"'H' cannot be part of an atom list and is removed.");else if(u==null)u=j(l.j7,H,6,1,15,1),u[0]=A;else {for(P=false,i=0;i<u.length;i++)if(A==u[i]){P=true;break;}if(!P){for(Q=j(l.j7,H,6,u.length+1,15,1),r=0;r<u.length;r++)Q[r]=u[r];Q[u.length]=A,u=Q;}}}return u!=null&&w.vzb(u,w.aAb(h0(y.vrb.prototype.kd,y.vrb,[]))),u;},E.vK=function(t,A){A.j==0?(E.xK(t),t.B=true,C.x2(t.v)):A.j==1?C.x2(t.v):F1(A.k)===F1(t.b)?A.a==1?(C.s3(t.D,""),C.k3(t.w,zr)):(C.s3(t.D,e.Il(t.A,t.a)),C.k3(t.w,FD)):F1(A.k)===F1(t.c)?(C.b2(t.f,false),C.o2(t.n,0),C.o2(t.j,0)):F1(A.k)===F1(t.f)&&C.b2(t.c,false);},E.wK=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B;for(L=e.Ol(t.A,t.a),pt(J(L,1),0)?(C.b2(t.b,true),C.k3(t.w,zr)):C.k3(t.w,FD),C.s3(t.D,e.Jl(t.A,t.a)==null?"":e.Kl(t.A,t.a)),A=J(L,Xn),pt(J(A,en),0)?C.o2(t.g,2):et(A,2)==0?C.o2(t.g,1):et(A,4)==0?C.o2(t.g,3):C.o2(t.g,0),b=J(L,120),et(b,112)==0?C.o2(t.s,1):et(b,96)==0?C.o2(t.s,2):et(b,8)==0?C.o2(t.s,3):et(b,104)==0?C.o2(t.s,4):et(b,88)==0?C.o2(t.s,5):et(b,56)==0?C.o2(t.s,6):C.o2(t.s,0),g=Jt(t0(J(L,kA),22)),C.o2(t.t,g==0?0:g-2),h=J(L,EA),u=-1,D=0;D<E.rK.length;D++)if(H1(h,E.rK[D])){u=D;break;}u!=-1?C.o2(t.r,u):(P=new T.Wjb("Custom:"),pt(J(h,QA),0)&&(P.a+=" 0"),pt(J(h,SA),0)&&(P.a+=" 3"),pt(J(h,VA),0)&&(P.a+=" 4"),pt(J(h,xA),0)&&(P.a+=" 5"),pt(J(h,zA),0)&&(P.a+=" 6"),pt(J(h,NA),0)&&(P.a+=" 7"),pt(J(h,De),0)&&(P.a+=" >=8"),t.C=h,C.j2(t.r,P.a),C.o2(t.r,E.rK.length)),o=J(L,aA),et(o,Pn)==0?C.o2(t.n,1):et(o,pe)==0?C.o2(t.n,2):et(o,Re)==0?C.o2(t.n,3):et(o,rn)==0?C.o2(t.n,4):et(o,v0)==0?C.o2(t.n,5):et(o,n0)==0?C.o2(t.n,6):et(o,te)==0?C.o2(t.n,7):et(o,iA)==0?C.o2(t.n,8):et(o,le)==0?C.o2(t.n,9):et(o,3276800)==0?C.o2(t.n,10):et(o,2228224)==0?C.o2(t.n,11):et(o,2490368)==0?C.o2(t.n,12):C.o2(t.n,0),r=J(L,Qe),et(r,zn)==0?C.o2(t.j,1):et(r,Nn)==0?C.o2(t.j,2):et(r,In)==0?C.o2(t.j,3):et(r,On)==0?C.o2(t.j,4):et(r,tr)==0?C.o2(t.j,5):et(r,Ar)==0?C.o2(t.j,6):et(r,fn)==0?C.o2(t.j,7):et(r,Ce)==0?C.o2(t.j,8):et(r,Gn)==0?C.o2(t.j,9):et(r,Jn)==0?C.o2(t.j,10):et(r,Dn)==0?C.o2(t.j,11):et(r,er)==0?C.o2(t.j,12):et(r,nr)==0?C.o2(t.j,13):et(r,Pr)==0?C.o2(t.j,14):C.o2(t.j,0),n=J(L,mA),et(n,nn)==0?C.o2(t.i,1):et(n,oe)==0?C.o2(t.i,2):et(n,ue)==0?C.o2(t.i,3):C.o2(t.i,0),i=J(L,X0),et(i,1792)==0?C.o2(t.k,1):et(i,1664)==0?C.o2(t.k,2):et(i,1408)==0?C.o2(t.k,3):et(i,128)==0?C.o2(t.k,4):et(i,384)==0?C.o2(t.k,5):et(i,896)==0?C.o2(t.k,6):et(i,1536)==0?C.o2(t.k,7):et(i,i0)==0?C.o2(t.k,8):et(i,1152)==0?C.o2(t.k,9):C.o2(t.k,0),Q=J(L,$A),et(Q,98304)==0?C.o2(t.p,1):et(Q,81920)==0?C.o2(t.p,2):et(Q,49152)==0?C.o2(t.p,3):et(Q,I0)==0?C.o2(t.p,4):C.o2(t.p,0),s=t0(J(L,ur),47),C.o2(t.o,Jt(s)),pt(J(L,j1),0)&&C.b2(t.c,true),pt(J(L,Z1),0)&&C.b2(t.f,true),B=J(L,Dr),H1(B,tn)?C.o2(t.u,1):H1(B,An)?C.o2(t.u,2):C.o2(t.u,0),pt(J(L,L0),0)&&C.b2(t.e,true),pt(J(L,r0),0)&&C.b2(t.d,true),t.q&&(c=J(L,TA),et(c,x0)==0?C.o2(t.q,1):et(c,Bn)==0?C.o2(t.q,2):et(c,TA)==0?C.o2(t.q,3):C.o2(t.q,0));},E.xK=function(t){var A,n;if(n=E.uK(t),e.Pm(t.A,t.a))for(A=0;A<t.A.q;A++)e.Pm(t.A,A)&&E.yK(t,A,n);else E.yK(t,t.a,n);e.Cq(t.A);},E.yK=function(t,A,n){var P,r,i,D,u,o;for(i=0,C.a2(t.b)?(i=rt(i,1),e.qn(t.A,A,n,true)):e.qn(t.A,A,n,false),C.k2(t.g)==2?e.aq(t.A,A)||(i=rt(i,Ur)):e.Tp(t.A,A)||(C.k2(t.g)==1?i=rt(i,2):C.k2(t.g)==3&&(i=rt(i,4))),u=0,r=0;r<e.$o(t.A,A);r++)pt(J(e.Ol(t.A,e.Zo(t.A,A,r)),r0),0)&&e.gq(t.A,e._o(t.A,A,r))&&++u;switch(C.k2(t.s)){case 1:u==0&&(i=rt(i,112));break;case 2:u<=2&&(i=rt(i,96));break;case 3:i=rt(i,8);break;case 4:u<3&&(i=rt(i,104));break;case 5:u<4&&(i=rt(i,88));break;case 6:i=rt(i,56);}switch(C.k2(t.t)!=0&&(i=rt(i,h1(Mt(C.k2(t.t)+2),22))),o=C.k2(t.r),o==E.rK.length?i=rt(i,t.C):o!=0&&(i=rt(i,E.rK[o])),C.k2(t.i)){case 1:i=rt(i,nn);break;case 2:i=rt(i,oe);break;case 3:i=rt(i,ue);}switch(D=e.up(t.A,A),C.k2(t.n)){case 1:D==1?i=rt(i,j1):D<1&&(i=rt(i,Pn));break;case 2:D==2?i=rt(i,j1):D<2&&(i=rt(i,pe));break;case 3:D==3?i=rt(i,j1):D<3&&(i=rt(i,Re));break;case 4:D==2?i=rt(i,j1):D<2&&(i=rt(i,rn));break;case 5:D==3?i=rt(i,j1):D<3&&(i=rt(i,v0));break;case 6:D==0&&(i=rt(i,Z1));break;case 7:D==1?i=rt(i,Z1):D<1&&(i=rt(i,te));break;case 8:D==2?i=rt(i,Z1):D<2&&(i=rt(i,iA));break;case 9:D==3?i=rt(i,Z1):D<3&&(i=rt(i,le));break;case 10:D==0?i=rt(i,3276800):i=rt(i,rn);break;case 11:D==0?i=rt(i,2228224):i=rt(i,v0);break;case 12:D<=1?i=rt(i,1099513856e3):D<=3&&(i=rt(i,v0));}switch(P=e.Po(t.A,A),C.k2(t.j)){case 1:P==0&&(i=rt(i,zn));break;case 2:P<=1&&(i=rt(i,Nn));break;case 3:P<=2&&(i=rt(i,In));break;case 4:P<=3&&(i=rt(i,On));break;case 5:P<2&&(i=rt(i,tr));break;case 6:P<3&&(i=rt(i,Ar));break;case 7:P<4&&(i=rt(i,fn));break;case 8:P==0&&(i=rt(i,Ce));break;case 9:P<2&&(i=rt(i,Gn));break;case 10:P<3&&(i=rt(i,Jn));break;case 11:P<4&&(i=rt(i,Dn));break;case 12:P<2&&(i=rt(i,er));break;case 13:P<3&&(i=rt(i,nr));break;case 14:P<3&&(i=rt(i,Pr));}switch(C.k2(t.k)){case 1:i=rt(i,1792);break;case 2:i=rt(i,1664);break;case 3:i=rt(i,1408);break;case 4:i=rt(i,128);break;case 5:i=rt(i,384);break;case 6:i=rt(i,896);break;case 7:i=rt(i,1536);break;case 8:i=rt(i,i0);break;case 9:i=rt(i,1152);}switch(C.k2(t.p)){case 1:i=rt(i,98304);break;case 2:i=rt(i,81920);break;case 3:i=rt(i,49152);break;case 4:i=rt(i,I0);}switch(C.k2(t.o)!=0&&(i=rt(i,h1(Mt(C.k2(t.o)),47))),C.a2(t.c)&&(e.jp(t.A,A)>0||e.Bl(t.A,A)==0&&(e.Tl(t.A,A)==5||e.bq(t.A,A)||e.Xp(t.A,A)))&&(i=rt(i,j1)),C.a2(t.f)&&(e.jp(t.A,A)>0||e.Bl(t.A,A)==0&&(e.Tl(t.A,A)==5||e.bq(t.A,A)||e.Xp(t.A,A)))&&(i=rt(i,Z1)),C.k2(t.u)){case 1:i=rt(i,tn);break;case 2:i=rt(i,An);}if(C.a2(t.e)&&(i=rt(i,L0)),C.a2(t.d)&&(i=rt(i,r0)),t.q)switch(C.k2(t.q)){case 1:i=rt(i,x0);break;case 2:i=rt(i,Bn);break;case 3:i=rt(i,TA);}e.vn(t.A,A,-1,false),e.vn(t.A,A,i,true);},E.zK=function(t){return t.B=false,C.A2(t.v),t.B;},E.AK=function(t,A,n,P){E.sK(),this.v=W.u1(t,(A.u[n]&512)!=0?"Atom Query Features (Multiple)":"Atom Query Features",this),E.tK(this,A,n,P);},N(292,1,se,E.AK),U.mc=function(t){E.vK(this,t);},U.a=0,U.B=false,U.C=0,l.S9=O(292),E.CK=function(t,A,n){var P,r,i,D;for(t.u=A,t.a=n,P=Rt(d.Math.round((d1.NN==-1&&(d1.NN=d1.PN()),d1.NN*8))),r=v(F(l.j7,1),H,6,15,[P,-1,-2,-2,P]),D=v(F(l.j7,1),H,6,15,[P,-2,P,-2,-2,-2,-2,-2,-2,-2,P,-2,P,-2,P,-2,P,-2,2*P,-2,P/2|0,-2,P/2|0,-2,2*P]),C.z2(t.s,r,D),C.s2(t.s,C.v2(t.s,"Desired Bond type(s):"),1,1,3,1),t.k=C.t2(t.s,"Single"),C.s2(t.s,t.k,1,3,3,3),t.c=C.t2(t.s,"Double"),C.s2(t.s,t.c,1,4,3,4),t.n=C.t2(t.s,"Triple"),C.s2(t.s,t.n,1,5,3,5),t.i=C.t2(t.s,"Quadruple"),C.s2(t.s,t.i,1,6,3,6),t.j=C.t2(t.s,"Quintuple"),C.s2(t.s,t.j,1,7,3,7),t.b=C.t2(t.s,"Delocalized"),C.s2(t.s,t.b,1,8,3,8),t.g=C.t2(t.s,"Coordinate (0-order)"),C.s2(t.s,t.g,1,9,3,9),t.q=C.u2(t.s),C.j2(t.q,"any ring state"),C.j2(t.q,BD),C.j2(t.q,"is any ring bond"),C.j2(t.q,"is non-aromatic ring bond"),C.j2(t.q,"is aromatic bond"),C.j2(t.q,"is any non-aromatic bond"),C.V1(t.q,t),C.s2(t.s,t.q,1,11,3,11),t.r=C.u2(t.s),C.j2(t.r,"any ring size"),C.j2(t.r,"is in 3-membered ring"),C.j2(t.r,"is in 4-membered ring"),C.j2(t.r,"is in 5-membered ring"),C.j2(t.r,"is in 6-membered ring"),C.j2(t.r,"is in 7-membered ring"),C.j2(t.r,"smallest ring 8 to 11"),C.j2(t.r,"smallest ring >= 12"),C.s2(t.s,t.r,1,13,3,13),t.e=C.t2(t.s,"Match formal bond order"),C.b2(t.e,(A.G[n]&Le)!=0),C.V1(t.e,t),C.s2(t.s,t.e,1,15,3,15),t.f=C.t2(t.s,"Match Stereo Configuration"),C.b2(t.f,(A.G[n]&hA)!=0),C.V1(t.f,t),C.s2(t.s,t.f,1,17,3,17),t.d=C.t2(t.s,"Is atom bridge between"),C.V1(t.d,t),C.s2(t.s,t.d,1,19,3,19),t.p=C.u2(t.s),i=0;i<16;i++)C.j2(t.p,""+i);C.r2(t.s,t.p,2,21),C.r2(t.s,C.v2(t.s," and"),3,21),C.V1(t.p,t),t.o=C.u2(t.s),E.GK(t,0),C.r2(t.s,t.o,2,23),C.r2(t.s,C.v2(t.s," atoms"),3,23),e.bw(t.u,7),E.HK(t);},E.DK=function(t){var A;A=C.a2(t.d),C.X1(t.k,!A),C.X1(t.c,!A),C.X1(t.n,!A),C.X1(t.i,!A),C.X1(t.j,!A),C.X1(t.b,!A),C.X1(t.g,!A),C.X1(t.f,!A&&e.fm(t.u,t.a)==2&&e.gm(t.u,t.a)!=0&&e.gm(t.u,t.a)!=3),C.X1(t.q,!A),C.X1(t.r,!A&&C.k2(t.q)!=1),C.X1(t.p,A),C.X1(t.o,A);},E.EK=function(t,A){var n,P,r;A.j==1?C.x2(t.s):A.j==0?(E.IK(t),t.v=true,C.x2(t.s)):F1(A.k)===F1(t.d)||F1(A.k)===F1(t.q)?E.DK(t):F1(A.k)===F1(t.p)&&(r=C.k2(t.p),t.t!=r&&(P=t.t+C.k2(t.o),n=E.GK(t,r),P<r?C.o2(t.o,0):P<r+n?C.o2(t.o,P-r):C.o2(t.o,n-1),t.t=r));},E.FK=function(t,A){return e.Pm(t.u,e.Zl(t.u,0,A))&&e.Pm(t.u,e.Zl(t.u,1,A));},E.GK=function(t,A){var n;for(C.m2(t.o),n=0;n<16;n++)C.j2(t.o,""+(A+n));return 16;},E.HK=function(t){var A,n,P,r,i,D,u;i=e.hm(t.u,t.a),P=e.im(t.u,t.a)==8||e.Yp(t.u,t.a)?6:e.fm(t.u,t.a),((i&1)!=0||P==1)&&C.b2(t.k,true),((i&2)!=0||P==2)&&C.b2(t.c,true),((i&4)!=0||P==3)&&C.b2(t.n,true),((i&32)!=0||P==4)&&C.b2(t.i,true),((i&64)!=0||P==5)&&C.b2(t.j,true),((i&8)!=0||P==6)&&C.b2(t.b,true),((i&16)!=0||P==0)&&C.b2(t.g,true),(i&Le)!=0&&C.b2(t.e,true),(i&hA)!=0&&C.b2(t.f,true),u=i&384,A=i&on,u==128?C.o2(t.q,1):A==v0?C.o2(t.q,4):u==256?C.o2(t.q,A==0?2:3):C.o2(t.q,A==0?0:5),D=(i&iA)>>17,C.o2(t.r,D==0?0:D<=2?D+5:D-2),(i&UA)!=0?(C.b2(t.d,true),r=(i&7680)>>9,n=(i&$n)>>13,C.o2(t.p,r),E.GK(t,r),C.o2(t.o,n)):C.o2(t.o,0),E.DK(t);},E.IK=function(t){var A;if(E.FK(t,t.a))for(A=0;A<t.u.r;A++)e.Pm(t.u,e.Zl(t.u,0,A))&&e.Pm(t.u,e.Zl(t.u,1,A))&&E.JK(t,A);else E.JK(t,t.a);e.Dq(t.u);},E.JK=function(t,A){var n,P,r,i,D;i=0,C.a2(t.d)?(r=C.k2(t.p),n=C.k2(t.o),i|=r<<9,i|=n<<13,i&=-128,e.Mn(t.u,A,1)):(C.a2(t.k)&&(i|=1),C.a2(t.c)&&(i|=2),C.a2(t.n)&&(i|=4),C.a2(t.i)&&(i|=32),C.a2(t.j)&&(i|=64),C.a2(t.b)&&(i|=8),C.a2(t.g)&&(i|=16),C.k2(t.q)!=0&&(C.k2(t.q)==1?e.gq(t.u,A)||(i|=128):C.k2(t.q)==2?e.gq(t.u,A)||(i|=256):C.k2(t.q)==3?e.Up(t.u,A)||(i|=4194560):C.k2(t.q)==4?e.Up(t.u,A)||(i|=v0):C.k2(t.q)==5&&(e.Up(t.u,A)||(i|=_0))),C.k2(t.r)!=0&&(D=C.k2(t.r)+2,D>7&&(D-=7),P=e.Yo(t.u,A),(D<=2||D!=P)&&(i|=D<<17)),C.a2(t.e)&&(i|=Le),C.a2(t.f)&&(i|=hA)),e.Ln(t.u,A,WA,false),e.Ln(t.u,A,i,true),e.Sk(t.u,A);},E.KK=function(t){return t.v=false,C.A2(t.s),t.v;},E.LK=function(t,A,n){this.s=W.u1(t,e.Pm(A,A.D[0][n])&&e.Pm(A,A.D[1][n])?"Bond Query Features (Multiple)":"Bond Query Features",this),E.CK(this,A,n);},N(293,1,se,E.LK),U.mc=function(t){E.EK(this,t);},U.a=0,U.t=0,U.v=false,l.T9=O(293),E.OK=function(){E.OK=D1,E.NK=v(F(l.Ybb,1),st,2,6,["None","One electron (duplet)","Two electrons (triplet)","Two electrons (singulet)"]);},E.PK=function(t){var A,n,P,r,i,D,u,o,Q;for(n=v(F(l.j7,1),H,6,15,[8,-2,8,-2,8]),Q=v(F(l.j7,1),H,6,15,[8,-2,4,-2,12,-2,4,-2,12,-2,4,-2,12,-2,8]),C.z2(t.c,n,Q),t.o=C.w2(t.c,1,1),C.V1(t.o,t),C.r2(t.c,C.v2(t.c,"Atom Label:"),1,1),C.r2(t.c,t.o,3,1),C.s2(t.c,C.v2(t.c,"(examples: 'D', 'Li', 'Cys', 'R12', 'R3@C')"),1,3,3,3),t.p=C.w2(t.c,1,1),C.V1(t.p,t),C.r2(t.c,C.v2(t.c,"Atom Mass:"),1,5),C.r2(t.c,t.p,3,5),C.s2(t.c,C.v2(t.c,"(empty for natural abundance)"),1,7,3,7),t.q=C.w2(t.c,1,1),C.V1(t.q,t),C.r2(t.c,C.v2(t.c,"Abnormal Valence:"),1,9),C.r2(t.c,t.q,3,9),C.s2(t.c,C.v2(t.c,"(empty for default valence)"),1,11,3,11),t.a==-1?(P=(e.Rk(),e.Lk)[t.k],C.s3(t.o,t.n==null?P:t.n+"@"+P),t.g!=0&&C.s3(t.p,""+t.g),t.j!=-1&&C.s3(t.q,""+t.j)):(P=e.Il(t.e,t.a),A=e.El(t.e,t.a),C.s3(t.o,A==null?P:A+"@"+P),e.Ml(t.e,t.a)!=0&&C.s3(t.p,""+e.Ml(t.e,t.a)),e.zl(t.e,t.a)!=-1&&C.s3(t.q,""+e.zl(t.e,t.a))),t.b=C.u2(t.c),i=E.NK,D=0,u=i.length;D<u;++D)r=i[D],C.j2(t.b,r);o=t.a==-1?t.i:e.Pl(t.e,t.a),C.o2(t.b,o==32?1:o==48?2:o==16?3:0),C.r2(t.c,C.v2(t.c,"Radical State:"),1,13),C.r2(t.c,t.b,3,13);},E.QK=function(t,A){A.j==1?(t.a!=-1&&(e.Cn(t.e,t.a,t.k),e.tn(t.e,t.a,t.g),e.gn(t.e,t.a,t.j),e.wn(t.e,t.a,t.i),e.mn(t.e,t.a,t.n)),C.x2(t.c)):A.j==0&&(E.RK(t),C.x2(t.c));},E.RK=function(t){var A,n,P,r,i,D,u;if(D=C.r3(t.o),n=null,w.aAb(D).length!=0&&(P=T.rjb(D,Y1(64)),P!=-1&&(n=(w.Wzb(0,P,w.aAb(D).length),w.aAb(D).substr(0,P)),D=(w.Xzb(P+1,w.aAb(D).length+1),w.aAb(D).substr(P+1)))),w.aAb(D).length!=0&&(A=e.po(D,t.d.c),A!=0||T.kjb(D,"?"))){if(r=0,w.aAb(C.r3(t.p)).length!=0)try{if(r=T.Whb(C.r3(t.p),P1,St),r<(e.Rk(),e.Pk)[A]-18||r>e.Pk[A]+12){C.B2(t.c,"Your mass is out of range!");return;}}catch(o){if(o=M1(o),Kt(o,40)){C.B2(t.c,"Your mass is not a number!");return;}else throw Ut(o);}if(u=-1,w.aAb(C.r3(t.q)).length!=0)try{if(u=T.Whb(C.r3(t.q),P1,St),u<0||u>15){C.B2(t.c,"Your valence is out of range!");return;}}catch(o){if(o=M1(o),Kt(o,40)){C.B2(t.c,"Your valence is not a number!");return;}else throw Ut(o);}i=C.k2(t.b)==1?32:C.k2(t.b)==2?48:C.k2(t.b)==3?16:0,E._L(t.d,A,r,u,i,n),t.a!=-1&&(e.fl(t.e,t.a,A,r,u,i),e.mn(t.e,t.a,n)),t.f=true,C.x2(t.c);}},E.SK=function(t){return t.f=false,C.A2(t.c),t.f;},E.TK=function(t,A,n,P,r,i,D){E.OK(),this.c=W.u1(t,vD,this),this.d=A,this.a=-1,this.k=n,this.g=P,this.j=r,this.i=i,this.n=D,E.PK(this);},E.UK=function(t,A,n,P){E.OK(),this.c=W.u1(t,vD,this),this.d=A,this.e=n,this.a=P,this.k=e.Tl(this.e,P),this.g=e.Ml(this.e,P),this.j=e.zl(this.e,P),this.i=e.Pl(this.e,P),this.n=e.El(this.e,P),E.PK(this);},N(220,1,se,E.TK,E.UK),U.mc=function(t){E.QK(this,t);},U.a=0,U.f=false,U.g=0,U.i=0,U.j=0,U.k=0,l.U9=O(220),y.WK=function(t){this.k=t;},N(317,1,{}),l.Ucb=O(317),Tt.XK=function(t,A){y.WK.call(this,A),this.j=t;},N(162,317,{}),U.j=0,l.cab=O(162),E.YK=function(t,A,n){Tt.XK.call(this,A,t),this.a=n;},N(24,162,{24:1},E.YK),U.a=false,l.V9=O(24),E.$K=function(t,A){y.sj(t.S,A);},E._K=function(t,A){var n,P,r,i;if(!A||A.q==0)return  false;if(A.r!=0)for(P=e.Vl(t.V,24*t.ab),e.ae(new Tt.bN(A),new C._2(W.J0(t.k)),new Tt.KN(0,0,W.H0(t.k),W.G0(t.k)),q1+Rt(P));E.dL(t,A,.2*P);)e._n(A,.5*P,.5*P);if(E.kM(t),e.Ym(A),e._m(A),r=t.V.K,t.V.q==0)e.ol(A,t.V),t.a||e.On(t.V,r),t.eb=d.Math.max(t.eb,4),W.M0(t.k),E.AL(t,new E.YK(t,1,true));else {for(i=t.V.q,e.Yk(t.V,A),t.a||e.On(t.V,r),n=0;n<t.V.q;n++)e.xn(t.V,n,n>=i);t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));}return  true;},E.aL=function(t,A){var n,P,r,i,D,u;for(D=(t.U&4)!=0,u=false,r=false,i=false,n=null,P=0;P<t.V.q;P++)if(e.Pm(t.V,P))if(u){if(i!=At.EH(n,e.Ql(t.V,P),e.Rl(t.V,P))){r=true;break;}}else {if(u=true,!D)break;n=y.xj(t.I,0),i=At.EH(n,e.Ql(t.V,P),e.Rl(t.V,P));}return A?D?r?(E.oL(t,true),true):u?(E.nL(t,true),false):(E.oL(t,false),true):(E.nL(t,u),false):D&&(r||!u);},E.bL=function(t){var A,n;e.bw(t.V,15),A=j(l.j7,H,6,t.V.q,15,1),n=e.hp(t.V,A,false,true),n=E.NL(t,A,n),E.jM(t,A,n),t.M=A,t.L=e.iw(t.V,A,n);},E.cL=function(t,A,n){var P,r,i,D,u;if(t.V.K){if(e.Jm(t.V,A)||e.Jm(t.V,n)||(P=e.Jl(t.V,A),r=e.Jl(t.V,n),P==null^r==null))return  false;if(P!=null){if(P.length!=r.length)return  false;for(i=0;i<P.length;i++)if(P[i]!=r[i])return  false;}if(D=pt(J(e.Ol(t.V,A),1),0),u=pt(J(e.Ol(t.V,n),1),0),D!=u)return  false;}return e.Tl(t.V,A)==e.Tl(t.V,n);},E.dL=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h;for(P=0,n*=n,u=0;u<A.q;u++){for(Q=A.J[u].a,s=A.J[u].b,D=false,o=0;o<t.V.q;o++)if(L=e.Ql(t.V,o),h=e.Rl(t.V,o),r=L-Q,i=h-s,r*r+i*i<n){D=true;break;}D&&++P;}return P==A.q;},E.eL=function(t){var A,n,P;for(E.bL(t),a.RI(new a.TI(),E.FL(t)),P=j(l.j7,H,6,t.L.length,15,1),A=0;A<t.V.q;A++)n=t.M[A],e.rn(t.V,A,e.Ll(t.L[n],P[n]),e.Ll(t.V,A)==0||e.Am(t.V,A)),++P[n];},E.fL=function(t,A){switch(A){case 0:E.mL(t);return;case 1:E.kM(t),t.eb=d.Math.max(t.eb,6),W.M0(t.k),E.AL(t,new E.YK(t,1,true));return;case 17:E.YL(t),t.eb=d.Math.max(t.eb,2),W.M0(t.k),E.AL(t,new E.YK(t,1,true));return;}},E.gL=function(t,A){var n,P,r,i;for(P=0,r=0,i=0,n=0;n<t.V.q;n++)(!A||e.Pm(t.V,n))&&(r+=e.Ql(t.V,n),i+=e.Rl(t.V,n),++P);return P>1?new j0.dgb(r/P,i/P):null;},E.hL=function(t,A,n){var P,r,i,D,u;for(u=j(l.kbb,i1,159,n,0,1),D=j(l.j7,H,6,n,15,1),i=0;i<n;i++)u[i]=new j0.cgb();for(P=0;P<t.V.q;P++)u[A[P]].a+=e.Ql(t.V,P),u[A[P]].b+=e.Rl(t.V,P),++D[A[P]];for(r=0;r<n;r++)u[r].a/=D[r],u[r].b/=D[r];return u;},E.iL=function(t,A){E.kM(t),e.$k(t.V,t.hb,t.kb,A,0,-1,0,null)&&(t.t!=-1&&(t.t=-1,E.AL(t,new E.YK(t,4,true))),t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));},E.jL=function(t,A){var n,P,r;for(P=0,n=0;n<t.V.q;n++)e.Pm(t.V,n)&&++P;r=P!=0&&P!=t.V.q,(t.U&1)!=0?E.lL(t,A,r):E.kL(t,A,r);},E.kL=function(t,A,n){var P;if(t.eb==6){if(n)for(P=0;P<t.V.q;P++)e.sn(t.V,P,!e.Pm(t.V,P));X.jA(new X.CA(n?4:0),t.V),n&&e.Zm(t.V);}e.Ak(t.G,A,new Tt.KN(0,0,W.H0(t.k),W.G0(t.k)),q1|t.T);},E.lL=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;if(n&&t.eb==6)for(B=j(l.j7,H,6,t.L.length,15,1),i=0;i<t.V.q;i++)h=t.M[i],e.sn(t.L[h],B[h],!e.Pm(t.V,i)),++B[h];for(u=j(l.hab,C1,35,t.L.length,0,1),b=0;b<t.L.length;b++)t.eb==6&&(X.jA(new X.CA(n?4:0),t.L[b]),e.yq(t.L[b])),o=new Tt.bN(t.L[b]),e.ae(o,A,null,q1|t.T),u[b]=o.v;for(M=1.8000000715255737*t.T,D=e.Vl(t.V,24*t.ab),P=(t.U&4)==0?0:t.eb==5?LD*W.H0(t.k):At.DH(y.xj(t.I,0))*t.T/D,k=.5*M,c=0;c<=t.L.length&&((t.U&4)!=0&&c==t.Y&&(At.FH(y.xj(t.I,0),k-M/2,W.G0(t.k)/2,k-M/2+P,W.G0(t.k)/2),k+=P),c!=t.L.length);c++)Q=k-u[c].c,L=.5*(W.G0(t.k)-u[c].a)-u[c].d,e._n(t.L[c],Q,L),k+=M+u[c].b;for(e.Ak(t.G,A,new Tt.KN(0,0,W.H0(t.k),W.G0(t.k)),q1|t.T),g=j(l.j7,H,6,t.L.length,15,1),r=0;r<t.V.q;r++)s=t.M[r],e.zn(t.V,r,e.Ql(t.L[s],g[s])),e.An(t.V,r,e.Rl(t.L[s],g[s])),++g[s];e.yq(t.V);},E.mL=function(t){var A;t.I&&(w.uzb(t.I.a,0),t.eb=d.Math.max(t.eb,1),W.M0(t.k)),E.kM(t),A=t.V.K,e._v(t.V),e.On(t.V,A),t.cb.q!=0?(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true))):(t.eb=d.Math.max(t.eb,1),W.M0(t.k));},E.nL=function(t,A){return t.V.q!=0&&t.r?C.d2(t.r,A?E.GL(t.V):e.fw(t.V)):false;},E.oL=function(t,A){var n;return n=A?E.HL(t):E.FL(t),n&&t.r?C.e2():false;},E.pL=function(t,A,n){var P;return e.rl(t.V,A,n)?(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true)),true):(P=E.yL(t,A,n),P?(y.Fj(t.I,P),t.v=null,t.eb=d.Math.max(t.eb,1),W.M0(t.k),true):false);},E.qL=function(t){return t.t!=-1?(e.ql(t.V,t.t),t.t=-1,t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true)),true):t.u!=-1?(e.ul(t.V,t.u),t.u=-1,t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true)),true):t.v&&!t.v.j?(y.Fj(t.I,t.v),t.v=null,t.eb=d.Math.max(t.eb,1),W.M0(t.k),true):false;},E.rL=function(t,A){var n,P,r;if(t.t!=-1&&(C.$2(A,(r=C.W2(A),r!=0?r:BP)),E.KL(t,A,t.t),t.w==19&&(P=e.Ll(t.V,t.t),P!=0)))for(n=0;n<t.V.f;n++)n!=t.t&&e.Ll(t.V,n)==P&&E.KL(t,A,n);t.u!=-1&&(C.$2(A,(r=C.W2(A),r!=0?r:BP)),E.LL(t,A,t.u)),t.v&&t.v.Mb(A);},E.sL=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c;for(i=0,n=0;n<t.V.q;n++)e.Pm(t.V,n)&&++i;for(b=t.V.q,c=t.V.r,t.gb=y.Iqb(t.gb,t.gb.length+i),t.jb=y.Iqb(t.jb,t.jb.length+i),D=j(l.j7,H,6,t.V.q,15,1),o=e.cn(t.V,1),Q=e.cn(t.V,2),P=0;P<b;P++)e.Pm(t.V,P)&&(s=t.V.q,t.gb[s]=t.gb[P],t.jb[s]=t.jb[P],D[P]=s,e.ll(t.V,t.V,P,o,Q));for(u=0;u<c;u++)e.Qm(t.V,u)&&e.nl(t.V,t.V,u,o,Q,D,false);for(r=0;r<b;r++)e.xn(t.V,r,false);for(A=b;A<t.V.q;A++)e.rn(t.V,A,0,false);if(t.I)for(L=t.I.a.length-1;L>=0;L--)h=y.xj(t.I,L),h.g&&!Kt(h,61)&&y.sj(t.I,h.Gb());},E.tL=function(t,A){var n,P;for(new E.UM(t._,A),P=false,n=0;n<w.aAb(A.e).length;n++)if(!e2(T.cjb(A.e,n))){P=true;break;}P||y.Fj(t.I,A),W.M0(t.k);},E.uL=function(t){var A,n,P;if(null.yd()||null.yd())E.aL(t,true);else if(null.yd())t.r&&C.g2();else if(null.yd())E.rM(t,null.yd());else if(null.yd())E.VL(t);else if(!null.yd())if(null.yd())E.eL(t),E.pM(t,d.Math.max(t.eb,1));else if(null.yd())E.XL(t,true,true);else if(null.yd())E.XL(t,false,true);else if(null.yd())E.XL(t,true,false);else if(null.yd())E.QL(t);else if(null.yd())E.ZK=!E.ZK,E.ZK&&E.eL(t);else if(null.yd())E.BL(t,true);else if(null.yd())E.BL(t,false);else if(null.yd())E.ZL(t);else if(null.yd())W.E1(t._,pD,RD);else if(null.yd()){if(null.yd(),A=T.Whb(null.yd(),P1,St),n=T.Whb(null.yd(),P1,St),e.Pm(t.V,A))for(P=0;P<t.V.f;P++)e.Pm(t.V,P)&&e.kn(t.V,P,n);else e.kn(t.V,A,n);}else null.yd()?(t.H=6,t.eb=d.Math.max(t.eb,1),W.M0(t.k)):null.yd()?(t.H=256,t.eb=d.Math.max(t.eb,1),W.M0(t.k)):null.yd()&&E.aM(t,t.w==19?16:0);},E.vL=function(t,A){var n,P,r,i,D,u,o,Q,L,s;if(!t.W){if(Q=A.b,A.j==1)if(Q==-3)t.Z=true,E.qM(t);else if(Q==-2)t.d=true,E.qM(t);else if(Q==-1)E.qM(t);else if(A.a)Q==122?(E.YL(t),t.eb=d.Math.max(t.eb,2),W.M0(t.k),E.AL(t,new E.YK(t,1,true))):Q==99?E.aL(t,true):Q==118&&((t.U&4)!=0&&t.r&&C.g2(),E.VL(t));else if(Q==-4)t.w!=19&&(E.kM(t),E.qL(t)||e.wl(t.V)&&(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true))));else if(Q==-6||t.t==-1&&Q==63){W.E1(t._,pD,RD);return;}else if(Q==-8)w.aAb(t.i.a).length!=0&&(E.xL(t,t.i.a),T.Ggb(t.i,0));else if(t.u!=-1)Q==113&&t.V.K?E.hM(t,t.u):Q==118?e.cl(t.V,t.u,3,false,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true))):Q>=52&&Q<=55?e.cl(t.V,t.u,Q-48,false,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true))):Q==97||Q==98?e.cl(t.V,t.u,6,true,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true))):(r=Q==48?(E.kM(t),e.il(t.V,t.u,16)):Q==49?(E.kM(t),e.il(t.V,t.u,1)):Q==50?(E.kM(t),e.il(t.V,t.u,2)):Q==51?(E.kM(t),e.il(t.V,t.u,4)):Q==117?(E.kM(t),e.il(t.V,t.u,257)):Q==100?(E.kM(t),e.il(t.V,t.u,129)):Q==99?(E.kM(t),e.il(t.V,t.u,386)):Q==109&&(E.kM(t),e.il(t.V,t.u,16)),r&&(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true))));else if(t.t!=-1){if(o=w.aAb(t.i.a).length==0,o?t.K=Q:(t.K==108&&(T.Ggb(t.i,0),T.Ojb(t.i,76)),t.K=-1),o&&Q==108)T.Sjb(t.i,"Cl"),t.eb=d.Math.max(t.eb,1),W.M0(t.k);else if(o&&(Q==43||Q==45))E.kM(t),e.hl(t.V,t.t,Q==43)&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));else if(o&&Q==46)E.kM(t),s=e.Pl(t.V,t.t)==32?0:32,e.wn(t.V,t.t,s),t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));else if(o&&Q==58)E.kM(t),s=e.Pl(t.V,t.t)==48?16:e.Pl(t.V,t.t)==16?0:48,e.wn(t.V,t.t,s),t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));else if(o&&Q==113&&t.V.K)E.gM(t,t.t);else if(o&&t.V.K&&(Q==120||Q==88))L=v(F(l.j7,1),H,6,15,[9,17,35,53]),e.pn(t.V,t.t,L),t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));else if(o&&Q==63)E.kM(t),e.fl(t.V,t.t,0,0,-1,0)&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));else if(o&&Q>48&&Q<=57){if(e.jp(t.V,t.t)>0){for(E.kM(t),i=Q-47,n=t.t,D=t.V.q-t.V.f,u=1;u<i&&(E.mM(t,n),P=e.Tk(t.V,t.ib,t.lb,0),P!=-1);u++)e.Vk(t.V,n,P),n=P-D,e.bw(t.V,1);t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));}}else !o&&Q==-7?(T.Ggb(t.i,0),t.eb=d.Math.max(t.eb,1),W.M0(t.k)):!o&&Q==-5?(T.Ggb(t.i,w.aAb(t.i.a).length-1),t.eb=d.Math.max(t.eb,1),W.M0(t.k)):Q>=65&&Q<=90||Q>=97&&Q<=122||Q>=48&&Q<=57||Q==45?(T.Ojb(t.i,Q&n1),t.eb=d.Math.max(t.eb,1),W.M0(t.k)):(Q==10||Q==13)&&E.xL(t,t.i.a);}else Q==104||Q==118?(t.U&7)==0&&E.BL(t,Q==104):t.db&&E.EM(t.db,A);A.j==2&&(Q==-3?(t.Z=false,E.qM(t)):Q==-2?(t.d=false,E.qM(t)):Q==-1?E.qM(t):t.db&&E.EM(t.db,A));}},E.wL=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M;if(A.j==1){if(t.t!=-1&&w.aAb(t.i.a).length!=0&&E.xL(t,t.i.a),T.Ggb(t.i,0),A.e){E.JL(t);return;}if(A.b==1){if(A.c==2)return;t.W=true,E.qM(t),E.SL(t,A);}}if(A.j==2){if(A.e){E.JL(t);return;}A.b==1&&(t.W=false,E.qM(t),E.TL(t));}if(A.j==3&&A.b==1&&A.c==2){E.IL(t,A.g,A.i);return;}if(A.j==4&&(W.A1(t._),E.qM(t)),A.j==6&&(t.W=false,k=A.g,M=A.i,E.oM(t,k,M,false)&&W.M0(t.k),E.qM(t)),A.j==7){switch(t.W=true,t.ib=A.g,t.lb=A.i,h=E.oM(t,t.ib,t.lb,true),t.X){case 2:E.lM(t)&&(h=true);break;case 1:(t.ib-t.hb)*(t.ib-t.hb)+(t.lb-t.kb)*(t.lb-t.kb)<100&&E.mM(t,t.e),h=true;break;case 3:e.zn(t.V,t.e,t.gb[t.e]+t.ib-t.hb),e.An(t.V,t.e,t.jb[t.e]+t.lb-t.kb),t.f!=-1&&(e.zn(t.V,t.f,t.gb[t.f]+t.ib-t.hb),e.An(t.V,t.f,t.jb[t.f]+t.lb-t.kb)),t.eb=d.Math.max(t.eb,2),W.M0(t.k);break;case 9:E.sL(t),t.X=4;case 4:if(t.I)for(i=new y.Bqb(t.I);i.a<i.c.a.length;)r=y.Aqb(i),r.g&&r.Ob(t.ib,t.lb);for(u=0;u<t.V.q;u++)e.Pm(t.V,u)&&(e.zn(t.V,u,t.gb[u]+t.ib-t.hb),e.An(t.V,u,t.jb[u]+t.lb-t.kb));t.eb=d.Math.max(t.eb,2),W.M0(t.k);break;case 11:y.sj(t.I,t.v.Gb()),t.X=10;case 10:t.v.Ob(t.ib,t.lb),t.eb=d.Math.max(t.eb,2),W.M0(t.k);break;case 7:for(b=false,P=0;P<t.V.q&&!b;P++)b=e.Pm(t.V,P);if(c=false,t.I)for(o=0;o<t.I.a.length&&!c;o++)c=y.xj(t.I,o).g;if(L=d.Math.abs(t.lb-t.kb)<20?1:d.Math.exp((t.lb-t.kb)/100),n=d.Math.abs(t.ib-t.hb)<20?0:(t.ib-t.hb)/50,g=b||c,t.I&&(!g||c)){for(o=0;o<t.I.a.length;o++)(!g||y.xj(t.I,o).g)&&y.xj(t.I,o).Pb(L,n);t.eb=d.Math.max(t.eb,2),W.M0(t.k);}(!g||b)&&(e.bo(t.V,L,n,g),t.eb=d.Math.max(t.eb,2),W.M0(t.k));break;case 5:case 6:if(B=null,t.X==5){if(d.Math.abs(t.ib-Tt.zN(t.R,t.R.a-1))<3&&d.Math.abs(t.lb-Tt.AN(t.R,t.R.a-1))<3)break;Tt.BN(t.R),Tt.yN(t.R,t.ib,t.lb),Tt.yN(t.R,t.hb,t.kb),B=t.R;}else B=new Tt.KN(d.Math.min(t.hb,t.ib),d.Math.min(t.kb,t.lb),d.Math.abs(t.ib-t.hb),d.Math.abs(t.lb-t.kb));if(t.I)for(o=0;o<t.I.a.length;o++)s=y.xj(t.I,o),Q=e.ke(s,B),(!t.O||!t.Q[o])&&Q!=s.g&&(s.g=Q,t.eb=d.Math.max(t.eb,1));for(D=0;D<t.V.q;D++)Q=B.Hb(Rt(e.Ql(t.V,D)),Rt(e.Rl(t.V,D))),(!t.O||!t.P[D])&&Q!=e.Pm(t.V,D)&&(e.xn(t.V,D,Q),t.eb=d.Math.max(t.eb,1));h=true;break;case 8:h=true;}h&&W.M0(t.k);}},E.xL=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c;if(T.Ggb(t.i,0),P=e.po(A,t.c),P!=0){if(E.kM(t),s=P==1^e.Tl(t.V,t.t)==1,e.fl(t.V,t.t,P,0,-1,0)){s&&(t.t=-1,E.AL(t,new E.YK(t,4,true))),t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));return;}t.eb=d.Math.max(t.eb,0),W.M0(t.k);}if(t.j!=null&&w.aAb(t.j).length!=0&&(A=t.j),b=(e.Rs(),L=(!e.Qs&&e.Ss(),y.opb(e.Qs,T.wjb(w.aAb(A).toLowerCase(),"-"))),L==null?null:e.Pq(new e._q(false),L)),b){for(E.kM(t),o=new e.tw(),e.xo(o,t.V,t.t,null),h=e.Vl(o,t.ab*24),D=o.q,n=0;n<o.q;n++)o.u[n]|=n0;for(e.dl(o,b,0),X.jA(new X.CA(4),o),r=e.Ql(t.V,t.t)-h*o.J[0].a,i=e.Rl(t.V,t.t)-h*o.J[0].b,u=t.V.q,e.dl(t.V,b,t.t),c=t.V.q-u,Q=0;Q<c;Q++)e.zn(t.V,u+Q,h*o.J[D+Q].a+r),e.An(t.V,u+Q,h*o.J[D+Q].b+i);e.yq(t.V),t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));}},E.yL=function(t,A,n){var P,r;if(t.I){for(r=new y.Bqb(t.I);r.a<r.c.a.length;)if(P=y.Aqb(r),P.Hb(A,n))return P;}return null;},E.zL=function(t){var A,n,P,r,i,D;for(i=-1,D=U0,A=0;A<t.V.q;A++)P=t.hb-e.Ql(t.V,A),r=t.kb-e.Rl(t.V,A),n=d.Math.sqrt(P*P+r*r),n<24&&D>n&&(D=n,i=t.M[A]);return i;},E.AL=function(t,A){var n;n=A.j,(n&t.J)==0&&(W.C1(t._,new E.vM(t,n,A)),t.J|=n);},E.BL=function(t,A){var n,P,r,i,D;for(D=false,P=0;P<t.V.q;P++)if(e.Pm(t.V,P)){D=true;break;}if(i=E.gL(t,D),i){for(E.kM(t),n=0;n<t.V.q;n++)(!D||e.Pm(t.V,n))&&(A?e.zn(t.V,n,2*i.a-e.Ql(t.V,n)):e.An(t.V,n,2*i.b-e.Rl(t.V,n)));for(r=0;r<t.V.r;r++)(!D||e.Qm(t.V,r))&&(e.im(t.V,r)==257?e.Mn(t.V,r,129):e.im(t.V,r)==129&&e.Mn(t.V,r,257));t.eb=d.Math.max(t.eb,1),W.M0(t.k);}},E.CL=function(t,A){return e.po(A,t.c)!=0?1:(t.j=e.Ts(A),t.j==null?E.ML(A)?3:4:w.aAb(t.j).length==0?3:2);},E.DL=function(t,A){var n,P,r;if(n=e.Zl(t.V,0,A),e.Nl(t.V,n)!=0)return e.ym(t.V,n)||e.Nl(t.V,n)!=1&&e.Nl(t.V,n)!=2?-1:n;if(e.Qo(t.V,n)==1){for(r=0;r<e.$o(t.V,n);r++)if(e.ap(t.V,n,r)==2&&(P=e.Zo(t.V,n,r),e.Qo(t.V,P)==2&&(e.Nl(t.V,P)==1||e.Nl(t.V,P)==2)))return P;}return  -1;},E.EL=function(t,A){var n;return n=e.Ho(t.V,e.Zl(t.V,0,A)),n!=-1&&e.gm(t.V,n)!=1&&e.gm(t.V,n)!=2&&(n=-1),n;},E.FL=function(t){var A,n;if((t.U&4)==0||t.L==null)return null;for(n=new At.zH(),A=0;A<t.L.length;A++)A<t.Y?At.dH(n,t.L[A]):At.bH(n,t.L[A]);return n;},E.GL=function(t){var A,n,P,r,i,D,u;for(P=0,n=0;n<t.q;n++)(t.u[n]&512)!=0&&++P;if(P==0)return null;for(i=0,r=0;r<t.r;r++)(t.u[t.D[0][r]]&t.u[t.D[1][r]]&512)!=0&&++i;for(u=j(l.Feb,ht,6,t.q,16,1),A=0;A<t.q;A++)u[A]=(t.u[A]&512)!=0;return D=new e.uw(P,i),e.Co(t,D,u,false,null),D;},E.HL=function(t){var A,n,P;for(n=new At.zH(),A=0;A<t.L.length;A++)P=E.GL(t.L[A]),P&&(A<t.Y?(y.sj(n.g,P),n.d=-1):(y.sj(n.f,P),n.d=-1));return n;},E.IL=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k;if(i=e.xl(t.V,A,n),D=e.yl(t.V,A,n),t.w==2){if(t.V.K){if(i!=-1)E.gM(t,i);else if(D!=-1)E.hM(t,D);else if(t.v){if(!t.Z){for(h=0;h<t.V.q;h++)e.xn(t.V,h,false);for(r=new y.Bqb(t.I);r.a<r.c.a.length;)P=y.Aqb(r),P.g=false;}e.ne(t.v,true),t.eb=d.Math.max(t.eb,1),W.M0(t.k);}}else if(k=-1,i!=-1?k=i:D!=-1&&(k=e.Zl(t.V,0,D)),k!=-1||t.v){if(!t.Z){for(h=0;h<t.V.q;h++)e.xn(t.V,h,false);if(t.I)for(Q=new y.Bqb(t.I);Q.a<Q.c.a.length;)o=y.Aqb(Q),o.g=false;}if(k!=-1){if((t.U&1)!=0)for(L=t.M[k],h=0;h<t.V.q;h++)t.M[h]==L&&e.xn(t.V,h,true);else for(s=e.fp(t.V,k,false),h=0;h<s.length;h++)e.xn(t.V,s[h],true);}else e.ne(t.v,true);t.eb=d.Math.max(t.eb,1),W.M0(t.k);}}else if(t.w==18&&(L=-2,(t.U&1)!=0&&(L=E.zL(t)),L!=-1)){for(B=St,g=P1,b=0;b<t.V.q;b++)(L==-2||t.M[b]==L)&&(B>e.Ql(t.V,b)&&(B=e.Ql(t.V,b)),g<e.Ql(t.V,b)&&(g=e.Ql(t.V,b)));if(g>B){for(u=(g+B)/2,c=0;c<t.V.q;c++)(L==-2||t.M[c]==L)&&e.zn(t.V,c,2*u-e.Ql(t.V,c));for(h=0;h<t.V.r;h++)if(L==-2||t.M[e.Zl(t.V,0,h)]==L)switch(e.im(t.V,h)){case 257:e.Mn(t.V,h,129);break;case 129:e.Mn(t.V,h,257);}}t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true));}},E.JL=function(t){t.r&&(E.aL(t,false),null.yd(),null.yd(),(t.U&4)!=0&&(null.yd(),null.yd()),(t.U&4)!=0&&null.yd()),(t.U&4)!=0&&t.w==19&&(null.yd(),null.yd(),null.yd(),null.yd(),null.yd(),null.yd()),t.w==18&&(null.yd(),null.yd(),null.yd()),t.g&&t.t!=-1&&(e.Cl(t.V,t.t),null.yd(),null.yd(),null.yd(),null.yd(),null.yd(),null.yd(),null.yd(),null.yd(),null.yd(),null.yd()),null.yd();},E.KL=function(t,A,n){var P,r,i;P=Rt(Rr*e.Ul(t.V)),r=Rt(e.Ql(t.V,n)),i=Rt(e.Rl(t.V,n)),C.O2(A,r-P,i-P,2*P);},E.LL=function(t,A,n){var P,r,i,D,u,o;r=Rt(Rr*e.Ul(t.V)),i=Rt(e.Ql(t.V,e.Zl(t.V,0,n))),u=Rt(e.Rl(t.V,e.Zl(t.V,0,n))),D=Rt(e.Ql(t.V,e.Zl(t.V,1,n))),o=Rt(e.Rl(t.V,e.Zl(t.V,1,n))),P=C.V2(A),C.Z2(A,r),C.K2(A,i,u,D,o),C.Z2(A,P);},E.ML=function(t){var A;if(w.aAb(t).length<3){for(A=1;A<(e.Rk(),e.Lk).length;A++)if(T.zjb(e.Lk[A],t))return  true;}return  false;},E.NL=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I;if(n<2)return n;for(S=j(l.Feb,i1,12,n,0,2),c=1;c<n;c++)S[c]=j(l.Feb,ht,6,c,16,1);for(D=e.Vl(t.V,24*t.ab),r=1;r<t.V.q;r++)for(i=0;i<r;i++)o=e.Ql(t.V,i)-e.Ql(t.V,r),Q=e.Rl(t.V,i)-e.Rl(t.V,r),u=d.Math.sqrt(o*o+Q*Q),u<bn*D&&(s=A[r],h=A[i],s!=h&&(s>h?S[s][h]=true:S[h][s]=true));for(I=j(l.j7,H,6,n,15,1),L=0;L<n;L++)I[L]=L;for(q=0,b=1;b<n;b++)for(k=0;k<b;k++)if(S[b][k]&&(g=I[b],B=I[k],g!=B))for(++q,V=d.Math.min(g,B),p=d.Math.max(g,B),M=0;M<n;M++)I[M]==p?I[M]=V:I[M]>p&&--I[M];for(P=0;P<t.V.q;P++)A[P]=I[A[P]];return n-q;},E.OL=function(t,A,n){var P,r;for(t.J&=~A,r=new y.Bqb(t.S);r.a<r.c.a.length;)P=y.Aqb(r),P.mc(n);},E.PL=function(t,A,n,P){return (t.U&6)!=0&&n[1]!=P[1]?n[1]==0?-1:1:JA(A[n[0]].a+A[n[0]].b,A[P[0]].a+A[P[0]].b);},E.QL=function(t){var A,n;for(n=false,A=0;A<t.V.f;A++)e.Ll(t.V,A)!=0&&e.Am(t.V,A)&&(n||(E.kM(t),n=true),e.rn(t.V,A,e.Ll(t.V,A),false));n&&E.pM(t,d.Math.max(t.eb,1));},E.RL=function(t){t.eb=d.Math.max(t.eb,4),W.M0(t.k),E.AL(t,new E.YK(t,1,false));},E.SL=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c;switch(t.hb=A.g,t.kb=A.i,t.w){case 18:if(b=t.hb,c=t.kb,t.e=e.xl(t.V,t.hb,t.kb),t.e!=-1&&(t.hb=e.Ql(t.V,t.e),t.kb=e.Rl(t.V,t.e)),e.co(t.V,b,c),t.I)for(u=new y.Bqb(t.I);u.a<u.c.a.length;)D=y.Aqb(u),D.Qb(b,c);E.kM(t),t.X=7;break;case 2:if(t.X=0,t.e=e.xl(t.V,t.hb,t.kb),t.e!=-1&&(t.f=-1,t.hb=e.Ql(t.V,t.e),t.kb=e.Rl(t.V,t.e),e.Pm(t.V,t.e)?t.X=t.Z?9:4:t.X=3),t.X==0&&(r=e.yl(t.V,t.hb,t.kb),r!=-1&&(t.e=e.Zl(t.V,0,r),t.f=e.Zl(t.V,1,r),e.Qm(t.V,r)?t.X=t.Z?9:4:t.X=3)),t.X==0&&t.v&&(t.v.g?t.X=t.Z?9:4:t.X=t.Z&&!Kt(t.v,61)?11:10),t.X!=0){for(t.gb=j(l.h7,Ft,6,t.V.q,15,1),t.jb=j(l.h7,Ft,6,t.V.q,15,1),o=0;o<t.V.q;o++)t.gb[o]=e.Ql(t.V,o),t.jb[o]=e.Rl(t.V,o);if(t.I)for(u=new y.Bqb(t.I);u.a<u.c.a.length;)D=y.Aqb(u),e.pe(D,t.hb,t.kb);E.kM(t);break;}if(A.a?t.X=6:(t.R=new Tt.CN(),Tt.yN(t.R,t.hb,t.kb),Tt.yN(t.R,t.hb,t.kb),t.X=5),t.P=j(l.Feb,ht,6,t.V.q,16,1),t.I&&(t.Q=j(l.Feb,ht,6,t.I.a.length,16,1)),t.O=A.f,t.O){for(Q=0;Q<t.V.q;Q++)t.P[Q]=e.Pm(t.V,Q);if(t.I)for(o=0;o<t.I.a.length;o++)t.Q[o]=y.xj(t.I,o).g;}else {for(s=false,Q=0;Q<t.V.q;Q++)e.Pm(t.V,Q)&&(e.xn(t.V,Q,false),s=true);if(t.I)for(o=0;o<t.I.a.length;o++)y.xj(t.I,o).g&&(e.ne(y.xj(t.I,o),false),s=true);s&&(t.eb=1,W.M0(t.k));}break;case 4:E.kM(t),E.pL(t,t.hb,t.kb);break;case 3:h=e.xl(t.V,t.hb,t.kb),h!=-1&&(E.kM(t),e.ln(t.V,h,!e.wm(t.V,h)),t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 101:case 103:case 102:t.u!=-1&&E.WL(t,t.u)&&(E.kM(t),E.bM(t,t.u,t.w==101?0:t.w==103?1:2),t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 5:case 6:case 23:if(t.e=e.xl(t.V,t.hb,t.kb),t.e==-1){if(P=e.yl(t.V,t.hb,t.kb),P!=-1){E.kM(t),i=511,t.w==6?i=257:t.w==23&&(i=129),e.il(t.V,P,i)&&(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;}}else {if(e.Mo(t.V,t.e)==8)return;t.hb=e.Ql(t.V,t.e),t.kb=e.Rl(t.V,t.e);}t.X=1,E.mM(t,t.e),W.M0(t.k);break;case 22:if(t.e=e.xl(t.V,t.hb,t.kb),t.e!=-1){if(e.Mo(t.V,t.e)==8)return;t.hb=e.Ql(t.V,t.e),t.kb=e.Rl(t.V,t.e);}t.X=2,t.q=0;break;case 7:E.kM(t),e.al(t.V,t.hb,t.kb,3,false,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 24:E.kM(t),e.al(t.V,t.hb,t.kb,4,false,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 8:E.kM(t),e.al(t.V,t.hb,t.kb,5,false,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 25:E.kM(t),e.al(t.V,t.hb,t.kb,6,false,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 9:E.kM(t),e.al(t.V,t.hb,t.kb,7,false,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 26:E.kM(t),e.al(t.V,t.hb,t.kb,6,true,e.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 10:E.kM(t),e.gl(t.V,t.hb,t.kb,true)&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 27:E.kM(t),e.gl(t.V,t.hb,t.kb,false)&&(t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));break;case 16:E.iL(t,1);break;case 11:E.iL(t,6);break;case 12:E.iL(t,7);break;case 13:E.iL(t,8);break;case 28:E.iL(t,14);break;case 29:E.iL(t,15);break;case 30:E.iL(t,16);break;case 14:E.iL(t,9);break;case 31:E.iL(t,17);break;case 15:E.iL(t,35);break;case 32:E.iL(t,53);break;case 33:A.d||A.a?(n=e.xl(t.V,t.hb,t.kb),n!=-1&&E.iM(t,n)):(E.kM(t),e.$k(t.V,t.hb,t.kb,t.F,t.B,t.D,t.C,t.A)&&(t.t=-1,t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true))));break;case 19:t.e=e.xl(t.V,t.hb,t.kb),t.e!=-1&&t.e<t.V.f&&(!t.V.K||!e.Jm(t.V,t.e))&&(t.hb=e.Ql(t.V,t.e),t.kb=e.Rl(t.V,t.e),t.X=8);break;case 21:L=null,t.v?Kt(t.v,94)&&(L=t.v):(L=new e.ax(),e.Ww(L,t.hb,t.kb),y.sj(t.I,L)),E.tL(t,L),E.kM(t),t.eb=d.Math.max(t.eb,3),W.M0(t.k);}},E.TL=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g;switch(h=t.X,t.X=0,h){case 1:if(c=e.xl(t.V,t.ib,t.lb),c!=-1&&e.Mo(t.V,c)==8)return;E.kM(t),t.e==-1&&(t.e=e.Tk(t.V,t.hb,t.kb,0)),c==-1&&(c=e.Tk(t.V,t.ib,t.lb,0)),t.e!=-1&&c!=-1&&(r=e.Lm(t.V,t.e)||e.Lm(t.V,c)?16:1,t.w==6&&(r=257),t.w==23&&(r=129),e._k(t.V,t.e,c,r)),t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));break;case 2:if(E.kM(t),t.q>0&&(t.e==-1&&(t.e=e.Tk(t.V,t.hb,t.kb,0)),t.n[0]==-1&&(t.n[0]=e.Tk(t.V,t.o[0],t.p[0],0)),t.n[0]!=-1&&e.Vk(t.V,t.e,t.n[0])),t.q>1)for(u=1;u<t.q;u++)t.n[u]==-1&&(t.n[u]=e.Tk(t.V,t.o[u],t.p[u],0)),t.n[u]!=-1&&e.Vk(t.V,t.n[u-1],t.n[u]);t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));break;case 3:case 4:case 7:t.eb=d.Math.max(t.eb,3),W.M0(t.k),E.AL(t,new E.YK(t,1,true));break;case 10:t.eb=d.Math.max(t.eb,3),W.M0(t.k);break;case 5:case 6:for(b=false,D=0;D<t.V.q;D++)if(t.P[D]!=e.Pm(t.V,D)){b=true;break;}W.M0(t.k),b&&E.AL(t,new E.YK(t,2,true));break;case 8:if(s=false,P=t.t,P!=-1&&t.V.K&&e.Jm(t.V,P)&&(P=-1),Q=e.Ll(t.V,t.e),P==-1){if(E.kM(t),Q!=0){for(s=true,g=false,A=0;A<t.V.f;A++)e.Ll(t.V,A)==Q&&(g=g|e.Am(t.V,A),e.rn(t.V,A,0,false));E.ZK&&!g&&E.eL(t);}}else {if(E.kM(t),s=true,g=false,t.e==P)for(o=e.Ll(t.V,t.e),A=0;A<t.V.f;A++)e.Ll(t.V,A)==o&&(g=g|e.Am(t.V,A),e.rn(t.V,A,0,false));else {for(L=e.Ll(t.V,P),n=0;n<t.V.f;n++)(e.Ll(t.V,n)==Q||e.Ll(t.V,n)==L)&&e.rn(t.V,n,0,false);for(i=1,A=0;A<t.V.f;A++)e.Ll(t.V,A)==i&&(++i,A=-1);e.rn(t.V,t.e,i,false),e.rn(t.V,P,i,false);}E.ZK&&!g&&E.eL(t);}s&&E.pM(t,d.Math.max(t.eb,1)),W.M0(t.k);}},E.UL=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q,S,V,I,Z,z,tt,ot,G,it,Qt,ut,Dt,lt,ct,yt,wt;if((t.fb!=W.H0(t.k)||t.N!=W.G0(t.k))&&(t.fb=W.H0(t.k),t.N=W.G0(t.k),t.eb<3&&(t.eb=3)),r=C.R2(A),u=C.U2(A),C.$2(A,r),C.Q2(A,0,0,t.fb,t.N),(t.U&4)!=0&&t.I.a.length==0&&(b=.5*t.fb,c=.5*t.N,i=.03999999910593033*t.fb,n=new At.GH(),At.FH(n,b-i,c,b+i,c),n.j=true,y.sj(t.I,n)),L=false,t.eb!=0){switch((t.U&1)!=0&&t.eb!=5&&E.bL(t),t.G=(t.U&4)!=0?new e.Dk(new At.BH(t.L,t.Y),t.I):(t.U&2)!=0?new e.Ek(t.L,t.Y,t.I):(t.U&1)!=0?new e.Fk(t.L,t.I):new e.Ck(t.V,t.I),e.yk(t.G,u,r),e.vk(t.G,t.T),e.zk(t.G,(t.U&1)==0?0:(s=dP,T.kjb(w.aAb(s).substr(0,18),xe)||T.kjb(w.aAb(s).substr(0,17),ze)||T.kjb(w.aAb(s).substr(0,14),Ne)?Lt.P3(r,Tn):(Lt.O3(),r&O0|Rt(d.Math.round(Tn*((r&W0)>>16)))<<16|Rt(d.Math.round(Tn*((r&Y0)>>8)))<<8|Rt(d.Math.round(Tn*(r&255)))))),e.xk(t.G,t.$),e.wk(t.G,t.H|8|(t.w==19?80:0)),t.eb){case 6:case 4:case 5:E.jL(t,A);break;case 3:S=e.Ak(t.G,A,new Tt.KN(0,0,t.fb,t.N),0),S&&(t.U&1)!=0&&e.vi(S,t.V);break;case 2:V=e.Bk(t.G,A,new Tt.KN(0,0,t.fb,t.N),0),L=!!V&&!(V.c==1&&V.a==0&&V.b==0);}t.eb=0;}switch(t.G&&e.tk(t.G,A),L||E.rL(t,A),t.G&&(e.uk(t.G,A),e.sk(t.G,A)),t.t!=-1&&w.aAb(t.i.a).length!=0&&(z=Rt(e.Ql(t.V,t.t)),ut=Rt(e.Rl(t.V,t.t)),B=t.i.a,I=E.CL(t,B),C.$2(A,I==1?u:I==2?-16776961:xn),I==1?B=(e.Rk(),e.Lk)[e.po(B,t.c)]:I==2&&(B=T.Bjb(t.j,0,w.aAb(B).length)),D=3*C.T2(A)/2|0,C.Y2(A,D,false,false),C.N2(A,z,ut,B),I==4&&(C.$2(A,fA),C.H2(A,z+C.S2(A,B).b/2,ut+D,"<unknown>")),I==2&&(C.$2(A,xn),C.N2(A,z+C.S2(A,B).b,ut,T.Ajb(t.j,w.aAb(B).length)))),C.$2(A,u),t.X){case 1:switch(tt=Rt(t.hb),Dt=Rt(t.kb),t.t==-1||t.t==t.e?(ot=Rt(t.ib),lt=Rt(t.lb)):(ot=Rt(e.Ql(t.V,t.t)),lt=Rt(e.Rl(t.V,t.t))),t.w){case 5:C.K2(A,tt,Dt,ot,lt);break;case 6:G=(Dt-lt)/9|0,ct=(ot-tt)/9|0,g=new Tt.DN(3),Tt.yN(g,tt,Dt),Tt.yN(g,ot+G,lt+ct),Tt.yN(g,ot-G,lt-ct),C.P2(A,new C.n3(g));break;case 23:for(G=ot-tt,ct=lt-Dt,Q=2;Q<17;Q+=2)it=tt+(Q*G/17|0)-(Q*ct/128|0),yt=Dt+(Q*ct/17|0)+(Q*G/128|0),Qt=tt+(Q*G/17|0)+(Q*ct/128|0),wt=Dt+(Q*ct/17|0)-(Q*G/128|0),C.K2(A,it,yt,Qt,wt);}break;case 2:if(t.q>0&&C.K2(A,Rt(t.hb),Rt(t.kb),Rt(t.o[0]),Rt(t.p[0])),t.q>1)for(Q=1;Q<t.q;Q++)C.K2(A,Rt(t.o[Q-1]),Rt(t.p[Q-1]),Rt(t.o[Q]),Rt(t.p[Q]));break;case 5:C.$2(A,(M=(p=C.W2(A),p!=0?p:BP),Lt.R3(M,(h=dP,T.kjb(w.aAb(h).substr(0,w.aAb(xe).length),xe)||T.kjb(w.aAb(h).substr(0,w.aAb(ze).length),ze)||T.kjb(w.aAb(h).substr(0,w.aAb(Ne).length),Ne)?CD:MD)))),C.L2(A,new C.n3(t.R)),C.$2(A,u);break;case 6:z=t.hb<t.ib?Rt(t.hb):Rt(t.ib),ut=t.kb<t.lb?Rt(t.kb):Rt(t.lb),Z=Rt(d.Math.abs(t.ib-t.hb)),o=Rt(d.Math.abs(t.lb-t.kb)),C.$2(A,(q=(k=C.W2(A),k!=0?k:BP),Lt.R3(q,(s=dP,T.kjb(w.aAb(s).substr(0,w.aAb(xe).length),xe)||T.kjb(w.aAb(s).substr(0,w.aAb(ze).length),ze)||T.kjb(w.aAb(s).substr(0,w.aAb(Ne).length),Ne)?CD:MD)))),C.M2(A,z,ut,Z,o),C.$2(A,u);break;case 8:tt=Rt(t.hb),Dt=Rt(t.kb),t.t==-1||t.t==t.e?(ot=Rt(t.ib),lt=Rt(t.lb)):(ot=Rt(e.Ql(t.V,t.t)),lt=Rt(e.Rl(t.V,t.t))),C.$2(A,(P=C.R2(A),Lt.T3(-8388608,P))),C.K2(A,tt,Dt,ot,lt),C.$2(A,u);}},E.VL=function(t){var A;return !!(t.r&&(A=C.f2(t.r),E._K(t,A)));},E.WL=function(t,A){return e.Rm(t.V,A)&&(E.DL(t,A)!=-1||E.EL(t,A)!=-1);},E.XL=function(t,A,n){var P,r;for(r=false,P=0;P<t.V.f;P++)e.Ll(t.V,P)!=0&&(n&&!e.Am(t.V,P)||A&&e.Am(t.V,P))&&(r||(E.kM(t),r=true),e.rn(t.V,P,0,false));r&&(A||E.eL(t),E.pM(t,d.Math.max(t.eb,1)));},E.YL=function(t){t.cb&&(e.ol(t.cb,t.V),t.I=t.bb?new e.gk(t.bb):null);},E.ZL=function(t){var A,n,P,r;for(r=false,n=0;n<t.V.q;n++)if(e.Pm(t.V,n)){r=true;break;}if(P=E.gL(t,r),P){for(E.kM(t),A=0;A<t.V.q;A++)(!r||e.Pm(t.V,A))&&(e.zn(t.V,A,2*P.a-e.Ql(t.V,A)),e.An(t.V,A,2*P.b-e.Rl(t.V,A)));t.eb=d.Math.max(t.eb,1),W.M0(t.k);}},E.$L=function(t,A){t.r=A;},E._L=function(t,A,n,P,r,i){t.F=A,t.B=n,t.D=P,t.C=r,t.A=i;},E.aM=function(t,A){t.H=A,t.eb=d.Math.max(t.eb,1),W.M0(t.k);},E.bM=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b,c;if(u=-1,r=E.DL(t,A),D=r==-1?E.EL(t,A):-1,n!=0){for(b=-1,L=0;L<t.V.f;L++)L!=r&&e.Hl(t.V,L)==n&&(!e.Qm(t.V,A)||!e.Pm(t.V,L))&&(o=e.Gl(t.V,L),b<o&&(b=o));for(s=0;s<t.V.g;s++)s!=D&&e.cm(t.V,s)==n&&(!e.Qm(t.V,A)||!e.Qm(t.V,s))&&(o=e.bm(t.V,s),b<o&&(b=o));if((r==-1?e.cm(t.V,D):e.Hl(t.V,r))!=n)u=d.Math.min(b+1,31);else if(u=r==-1?e.bm(t.V,D):e.Gl(t.V,r),e.Qm(t.V,A)){for(c=true,h=0;h<t.V.f;h++)if(h!=r&&e.Pm(t.V,h)&&e.Hl(t.V,h)==n&&e.Gl(t.V,h)!=u){c=false;break;}for(Q=0;Q<t.V.g;Q++)if(Q!=D&&e.Qm(t.V,Q)&&e.cm(t.V,Q)==n&&e.bm(t.V,Q)!=u){c=false;break;}c&&(u<=b?(++u,u==32&&(u=0)):u=0);}else u<=b?(++u,u==32&&(u=0)):u=0;}if(e.Qm(t.V,A))for(Q=0;Q<t.V.g;Q++)e.Qm(t.V,Q)&&e.Rm(t.V,Q)&&(P=E.DL(t,Q),i=E.EL(t,Q),P!=-1?e.on(t.V,P,n,u):i!=-1&&e.Gn(t.V,i,n,u));else r!=-1?e.on(t.V,r,n,u):D!=-1&&e.Gn(t.V,D,n,u);},E.cM=function(t,A){t.V!=A&&(E.kM(t),t.V=A,t.U=0,t.I=null,t.eb=d.Math.max(t.eb,4),W.M0(t.k),E.AL(t,new E.YK(t,1,false)));},E.dM=function(t,A){var n,P,r,i;for(e._v(t.V),t.L=j(l.j8,OA,26,A.g.a.length+A.f.a.length,0,1),t.Y=A.g.a.length,r=0;r<A.g.a.length+A.f.a.length;r++)t.L[r]=r<A.g.a.length?y.xj(A.g,r):y.xj(A.f,r-A.g.a.length),e.Yk(t.V,t.L[r]);for(e.On(t.V,A.c||At.gH(A)),E.kM(t),t.M=j(l.j7,H,6,t.V.q,15,1),n=0,P=0;P<t.L.length;P++)for(i=0;i<t.L[P].q;i++)t.M[n++]=P;t.I=new e.fk(),t.U=5,t.eb=d.Math.max(t.eb,5),W.M0(t.k),E.AL(t,new E.YK(t,1,false));},E.eM=function(t,A){t.db=A;},E.fM=function(t,A,n){var P;return P=y.xj(t.I,0),At.EH(P,e.Ql(t.V,A),e.Rl(t.V,A))==At.EH(P,e.Ql(t.V,n),e.Rl(t.V,n));},E.gM=function(t,A){var n,P;t.b&&(E.kM(t),P=(t.U&4)!=0,n=new E.AK(t._,t.V,A,P),E.zK(n)&&(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true))));},E.hM=function(t,A){var n;t.b&&(E.kM(t),n=new E.LK(t._,t.V,A),E.KK(n)&&(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true))));},E.iM=function(t,A){var n;E.kM(t),n=A==-1?new E.TK(t._,t,t.F,t.B,t.D,t.C,t.A):new E.UK(t._,t,t.V,A),E.SK(n)&&A!=-1&&(t.eb=d.Math.max(t.eb,1),W.M0(t.k),E.AL(t,new E.YK(t,1,true)));},E.jM=function(t,A,n){var P,r,i,D,u,o,Q,L,s,h,b;for(s=y1(l.j7,[E1,H],[7,6],15,[n,(t.U&6)!=0?2:1],2),o=0;o<n;o++)s[o][0]=o;if(L=E.hL(t,A,n),(t.U&4)!=0)for(t.Y=0,P=(t.U&4)!=0?y.xj(t.I,0):null,Q=0;Q<n;Q++)s[Q][1]=At.EH(P,L[Q].a,L[Q].b)?1:0,s[Q][1]==0&&++t.Y;else if((t.U&2)!=0)for(t.Y=n,r=0;r<t.V.q;r++)e.Tl(t.V,r)==0&&s[A[r]][1]==0&&(s[A[r]][1]=1,--t.Y);for(D=L,y.drb(s,new E.xM(t,D)),h=j(l.j7,H,6,n,15,1),u=0;u<n;u++)b=s[u][0],h[b]=u;for(i=0;i<t.V.q;i++)A[i]=h[A[i]];},E.kM=function(t){!t.cb&&(t.cb=new e.eo()),e.ol(t.V,t.cb),t.bb=t.I?new e.gk(t.I):null;},E.lM=function(t){var A,n,P,r,i,D,u,o,Q,L,s,h,b,c,g,B,k,M,p,q;if(p=e.no(t.hb,t.kb,t.ib,t.lb),k=t.ib-t.hb,M=t.lb-t.kb,s=t.q,h=0,c=0,b=0,g=0,s>0&&(h=Jt(c1(d.Math.round(t.o[0]))),c=Jt(c1(d.Math.round(t.p[0])))),s>1&&(b=Jt(c1(d.Math.round(t.o[1]))),g=Jt(c1(d.Math.round(t.p[1])))),o=0,t.e==-1||e.Mo(t.V,t.e)==0)o=b0*ee(c1(d.Math.round(p*3/A1)));else if(e.Mo(t.V,t.e)==1)n=e.Yl(t.V,e.Zo(t.V,t.e,0),t.e),P=n-b0,r=n+b0,o=d.Math.abs(e.oo(p,P))<d.Math.abs(e.oo(p,r))?P:r;else {for(i=j(l.h7,Ft,6,e.Mo(t.V,t.e),15,1),L=0;L<e.Mo(t.V,t.e);L++)i[L]=e.Yl(t.V,t.e,e.Zo(t.V,t.e,L));for(w.vzb(i,w.aAb(h0(y.frb.prototype.kd,y.frb,[]))),Q=0;Q<i.length;Q++){if(B=Q==0?i[i.length-1]-I1:i[Q-1],B<p&&p<i[Q]){o=(i[Q]+B)/2;break;}if(B<p-I1&&p-I1<i[Q]){o=(i[Q]+B)/2;break;}}}if(A=e.Vl(t.V,24*t.ab),t.q=d.Math.abs(e.oo(p,o))>b0?0:Rt(d.Math.sqrt(k*k+M*M)/A),t.q>0){for((t.o==null||t.o.length<t.q)&&(t.o=j(l.h7,Ft,6,t.q,15,1),t.p=j(l.h7,Ft,6,t.q,15,1)),D=j(l.h7,Ft,6,2,15,1),u=j(l.h7,Ft,6,2,15,1),q=e.oo(p,o)<0?o-b0:o+b0,D[0]=A*d.Math.sin(o),u[0]=A*d.Math.cos(o),D[1]=A*d.Math.sin(q),u[1]=A*d.Math.cos(q),L=0;L<t.q;L++)t.o[L]=(L==0?t.hb:t.o[L-1])+D[L&1],t.p[L]=(L==0?t.kb:t.p[L-1])+u[L&1];for(t.n=j(l.j7,H,6,t.q,15,1),Q=0;Q<t.q;Q++)t.n[Q]=e.xl(t.V,t.o[Q],t.p[Q]),t.n[Q]!=-1&&(t.o[Q]=e.Ql(t.V,t.n[Q]),t.p[Q]=e.Rl(t.V,t.n[Q]));}return s!=t.q||t.q!=0&&(h!=Jt(c1(d.Math.round(t.o[0])))||c!=Jt(c1(d.Math.round(t.p[0]))))||t.q>1&&(b!=Jt(c1(d.Math.round(t.o[1])))||g!=Jt(c1(d.Math.round(t.p[1]))));},E.mM=function(t,A){var n,P,r,i,D,u,o,Q,L,s,h;if(s=Ve,A!=-1){for(n=j(l.h7,Ft,6,9,15,1),D=0;D<e.Mo(t.V,A);D++)n[D]=e.Yl(t.V,A,e.Zo(t.V,A,D));if(e.Mo(t.V,A)==1)n[0]<-2.6179938779914944?s=b0:n[0]<un?s=Ve:n[0]<Sr?s=b0:n[0]<0?s=Ve:n[0]<Me?s=rr:n[0]<k0?s=cn:n[0]<2.6179938779914944?s=rr:s=cn;else {for(u=e.Mo(t.V,A)-1;u>0;u--)for(o=0;o<u;o++)n[o]>n[o+1]&&(h=n[o],n[o]=n[o+1],n[o+1]=h);for(n[e.Mo(t.V,A)]=n[0]+I1,L=0,Q=0,i=0;i<e.Mo(t.V,A);i++)P=n[i+1]-n[i],Q<P&&(Q=P,L=i);s=(n[L]+n[L+1])/2;}}r=e.Vl(t.V,24*t.ab),t.ib=(A==-1?t.hb:e.Ql(t.V,A))+r*d.Math.sin(s),t.lb=(A==-1?t.kb:e.Rl(t.V,A))+r*d.Math.cos(s);},E.nM=function(t,A){t.w!=A&&(t.w==19||A==19?(t.w=A,t.eb=d.Math.max(t.eb,1),W.M0(t.k)):t.w=A);},E.oM=function(t,A,n,P){var r,i,D,u,o,Q;if(D=e.xl(t.V,A,n),u=-1,P&&t.X==8&&D!=-1&&(!E.cL(t,t.e,D)||e.Ll(t.V,t.e)!=0&&e.Ll(t.V,t.e)==e.Ll(t.V,D)&&!e.Am(t.V,t.e)||E.fM(t,t.e,D))&&(D=-1),D!=-1&&(t.w==101||t.w==103||t.w==102?(u=e.Ip(t.V,D),D=-1):t.w==19&&D>=t.V.f&&(D=-1)),u==-1&&D==-1&&t.w!=22&&t.w!=19&&t.w!=3&&t.w!=10&&t.w!=27&&t.w!=16&&t.w!=11&&t.w!=12&&t.w!=13&&t.w!=28&&t.w!=29&&t.w!=30&&t.w!=14&&t.w!=31&&t.w!=15&&t.w!=32&&t.w!=33&&(u=e.yl(t.V,A,n)),u!=-1&&(t.w==101||t.w==103||t.w==102)&&!(e.Rm(t.V,u)&&(E.DL(t,u)!=-1||E.EL(t,u)!=-1))&&(u=-1),r=t.v,!P&&t.I&&(r=null,D==-1&&u==-1&&(t.w==2||t.w==4||t.w==21))){for(Q=new y.Bqb(t.I);Q.a<Q.c.a.length;)if(o=y.Aqb(Q),(t.w==2||t.w==4&&!Kt(o,61)||t.w==21&&Kt(o,94))&&o.Eb(A,n)){r=o,t.v&&t.v!=o&&t.v.Fb();break;}}return i=t.t!=D||t.u!=u||t.v!=r||!!r,t.t!=D&&(t.t!=-1&&w.aAb(t.i.a).length!=0?E.xL(t,t.i.a):(t.t=D,T.Ggb(t.i,0),E.AL(t,new E.YK(t,4,true)))),t.u!=u&&(t.u=u,E.AL(t,new E.YK(t,8,true))),t.v=r,i;},E.pM=function(t,A){t.eb=d.Math.max(t.eb,A),W.M0(t.k),E.AL(t,new E.YK(t,1,true));},E.qM=function(t){var A;switch(A=-1,t.w){case 18:A=9;break;case 2:t.t!=-1&&e.Pm(t.V,t.t)||t.u!=-1&&e.Qm(t.V,t.u)?A=t.W?4:t.Z?3:2:t.t!=-1||t.u!=-1?A=11:t.v?A=t.W?4:t.Z&&!Kt(t.v,61)?3:2:A=t.Z?t.d?8:6:t.d?7:5;break;case 4:A=1;break;case 22:A=0;break;case 21:A=12;break;default:A=11;}t.s!=A&&(t.s=A,W.D1(t._,A));},E.rM=function(t,A){E.kM(t),E.dM(t,At.UH(A,true,null));},E.sM=function(t,A,n,P){this.V=t,this.U=A,this._=n,this.k=P,this.S=new y.Mj(),this.w=5,this.t=-1,this.u=-1,this.v=null,this.e=-1,this.F=6,this.B=0,this.D=-1,this.C=0,this.A=null,this.b=true,this.a=false,this.X=0,this.s=11,this.i=new T.Ujb(),this.c=331,this.$=1,this.ab=(d1.NN==-1&&(d1.NN=d1.PN()),d1.NN),this.T=Rt(d.Math.round((d1.NN==-1&&(d1.NN=d1.PN()),d1.NN*24))),(this.U&6)!=0&&(this.U|=1),(this.U&12)!=0&&(this.I=new e.fk()),this.eb=4;},N(286,1,se,E.sM),U.mc=function(t){var A;Kt(t,140)?E.uL((A=this,A)):Kt(t,156)?E.vL(this,t):Kt(t,137)&&E.wL(this,t);},U.a=false,U.b=false,U.c=0,U.d=false,U.e=0,U.f=0,U.g=false,U.q=0,U.s=0,U.t=0,U.u=0,U.w=0,U.B=0,U.C=0,U.D=0,U.F=0,U.H=0,U.J=0,U.K=0,U.N=0,U.O=false,U.T=0,U.U=0,U.W=false,U.X=0,U.Y=0,U.Z=false,U.$=0,U.ab=0,U.eb=0,U.fb=0,U.hb=0,U.ib=0,U.kb=0,U.lb=0,E.ZK=true,l.Y9=O(286),E.vM=function(t,A,n){this.a=t,this.c=A,this.b=n;},N(296,1,{262:1},E.vM),U.nc=function(){E.OL(this.a,this.c,this.b);},U.c=0,l.W9=O(296),E.xM=function(t,A){this.a=t,this.b=A;},N(297,1,{},E.xM),U.Rb=function(t,A){return E.PL(this.a,this.b,t,A);},U.mb=function(t){return this===t;},l.X9=O(297),E.CM=function(){E.CM=D1,E.AM=Rt(d.Math.round(Rt(d.Math.round((d1.NN==-1&&(d1.NN=d1.PN()),d1.NN*2))))),E.BM=(d1.NN==-1&&(d1.NN=d1.PN()),d1.NN*21);},E.DM=function(t,A,n,P,r){var i;P|=P1,i=E.HM(n),P!=-1&&(C.$2(A,P),C.Q2(A,i[0],i[1],E.BM,E.BM)),n==20?C.J2(A,t.c,0,t.d*84,84,84,i[0],i[1],E.BM,E.BM):C.J2(A,r?t.g:t.i,(i[0]-E.AM)*t.j,(i[1]-E.AM)*t.j,84,84,i[0],i[1],E.BM,E.BM);},E.EM=function(t,A){Kt(A,137)?E.GM(t,A):Kt(A,156)&&E.FM(t,A);},E.FM=function(t,A){var n;n=A.b,A.j==1&&(n==-9&&t.b>=17?E.LM(t,t.b-17):n==-10&&t.b<17?E.LM(t,t.b+17):n==-11&&t.b%17>0?E.LM(t,t.b-1):n==-12&&t.b%17<16?E.LM(t,t.b+1):n==122?E.LM(t,18):n==32?E.LM(t,2):n==48?E.LM(t,4):n==116?E.LM(t,21):n==49?E.LM(t,5):n==50?E.LM(t,22):n==109?E.LM(t,19):n==51?E.LM(t,7):n==52?E.LM(t,24):n==53?E.LM(t,8):n==54?E.LM(t,25):n==55?E.LM(t,9):n==97?E.LM(t,26):n==45?E.LM(t,27):n==43?E.LM(t,10):n==117?E.LM(t,6):n==100?E.LM(t,23):n==99?E.LM(t,11):n==110?E.LM(t,12):n==112?E.LM(t,29):n==111?E.LM(t,13):n==115?E.LM(t,30):n==102?E.LM(t,14):n==108?E.LM(t,31):n==98?E.LM(t,15):n==105?E.LM(t,32):n==104?E.LM(t,16):n==46&&E.LM(t,33));},E.GM=function(t,A){var n,P;if(A.j==1){if(n=E.IM(t,A),!(n>=0&&n<34&&(n!=t.b||n==20||n==33)&&(n!=19||(t.a.U&4)!=0)&&(n!=21||(t.a.U&8)!=0)))return;n==20&&n==t.b&&(t.d=++t.d%3),t.k=n,W.k1(t.n);}else if(A.j==2){if(t.k==-1)return;if(P=E.IM(t,A),P!=t.k||t.k==19&&(t.a.U&4)==0||t.k==21&&(t.a.U&8)==0){t.k=-1,W.k1(t.n);return;}if(t.k=-1,P==0||P==1||P==17){W.k1(t.n),E.fL(t.a,P);return;}t.b=P,W.k1(t.n),t.b==20?E.nM(t.a,t.d==0?101:t.d==1?103:102):(t.b==33&&E.iM(t.a,-1),E.nM(t.a,P));}else (A.j==6||A.j==5)&&(n=E.IM(t,A),n==t.k&&(n=-1),n!=t.f&&(t.f=n,W.k1(t.n)));},E.HM=function(t){var A;return A=j(l.h7,Ft,6,2,15,1),A[0]=E.BM*(t/17|0)+E.AM,A[1]=E.BM*(t%17)+E.AM,A;},E.IM=function(t,A){var n,P,r;return P=A.g-E.AM,r=A.i-E.AM,P<0||P>=2*E.BM||r<0||r>=17*E.BM?-1:(n=17*Rt(P/E.BM)+Rt(r/E.BM),n>=0&&n<34&&(n!=t.b||n==20||n==33)&&(n!=19||(t.a.U&4)!=0)&&(n!=21||(t.a.U&8)!=0)?n:-1);},E.JM=function(t){t.i=W.w1(t.a._,Jr),d1.UN(t.i),t.g=W.w1(t.a._,Jr),d1.ON(t.g),t.o=2*E.AM+d1.QN(C.e3(t.i)/4|0),t.e=2*E.AM+d1.QN(C.b3(t.i)/4|0),t.c=W.w1(t.a._,HD),t.j=(d1.NN==-1&&(d1.NN=d1.PN()),4/d1.NN),t.b=5,t.k=-1,t.f=-1,E.eM(t.a,t);},E.KM=function(t,A){var n,P,r,i,D,u,o;n=W.g1(t.n),r=(Lt.O3(),(Lt.N3[0]*((n&W0)>>16)+Lt.N3[1]*((n&Y0)>>8)+Lt.N3[2]*(n&255))/255<.5),P=r?Lt.P3(n,fe):n&O0|Rt(d.Math.round(sn*((n&W0)>>16)))<<16|Rt(d.Math.round(sn*((n&Y0)>>8)))<<8|Rt(d.Math.round(sn*(n&255))),D=r?Lt.P3(n,.5):n&O0|Rt(d.Math.round(he*((n&W0)>>16)))<<16|Rt(d.Math.round(he*((n&Y0)>>8)))<<8|Rt(d.Math.round(he*(n&255))),o=C.e3(t.i),u=C.b3(t.i),C.J2(A,t.i,0,0,o,u,E.AM,E.AM,o/t.j,u/t.j),i=E.HM(20),C.J2(A,t.c,0,t.d*84,84,84,i[0],i[1],E.BM,E.BM),(t.a.U&4)==0&&E.DM(t,A,19,-1,true),(t.a.U&8)==0&&E.DM(t,A,21,-1,true),E.DM(t,A,t.b,D,false),t.f!=-1&&t.f!=t.k&&E.DM(t,A,t.f,P,false),t.k!=-1&&E.DM(t,A,t.k,7167924,false);},E.LM=function(t,A){t.b!=A&&A>=0&&A<34&&(A!=t.b||A==20||A==33)&&(A!=19||(t.a.U&4)!=0)&&(A!=21||(t.a.U&8)!=0)&&!(A==0||A==17||A==1)&&(t.b=A,E.nM(t.a,A),W.k1(t.n));},E.MM=function(t,A){E.CM(),this.n=t,this.a=A,E.JM(this);},N(298,1,se,E.MM),U.mc=function(t){E.EM(this,t);},U.b=0,U.d=0,U.e=0,U.f=0,U.j=0,U.k=0,U.o=0,E.AM=0,E.BM=0,l.Z9=O(298),E.RM=function(){E.RM=D1,E.PM=v(F(l.j7,1),H,6,15,[0,2,1,3]),E.QM=v(F(l.Ybb,1),st,2,6,["plain","italic","bold","italics & bold"]),E.OM=v(F(l.Ybb,1),st,2,6,["8","9","10","12","14","18","24","32"]);},E.SM=function(t){var A,n,P,r,i,D,u,o,Q,L,s;for(t.b=C.u2(t.c),i=E.OM,u=0,Q=i.length;u<Q;++u)P=i[u],C.j2(t.b,P);for(C.n2(t.b,true),C.p2(t.b,""+Rt(t.e.c)),t.a=C.u2(t.c),r=E.QM,D=0,o=r.length;D<o;++D)P=r[D],C.j2(t.a,P);for(L=0,n=0;n<E.PM.length;n++)if(t.e.d==E.PM[n]){L=n;break;}C.o2(t.a,L),A=v(F(l.j7,1),H,6,15,[8,-2,4,-2,8]),s=v(F(l.j7,1),H,6,15,[8,-2,4,-2,8,-2,8]),C.z2(t.c,A,s),C.r2(t.c,C.v2(t.c,"Text size:"),1,1),C.r2(t.c,t.b,3,1),C.r2(t.c,C.v2(t.c,"Text style:"),1,3),C.r2(t.c,t.a,3,3),t.d=C.w2(t.c,20,3),C.s3(t.d,t.e.e),C.s2(t.c,t.d,1,5,3,5);},E.TM=function(t,A){var n,P;if(A.j==0){try{n=T.Whb(C.l2(t.b),P1,St);}catch(r){if(r=M1(r),Kt(r,40)){C.B2(t.c,"Illegal text size.");return;}else throw Ut(r);}P=E.PM[C.k2(t.a)],e.Zw(t.e,C.r3(t.d),n,P);}C.x2(t.c);},E.UM=function(t,A){E.RM(),this.c=W.u1(t,"Edit Text",this),this.e=A,E.SM(this);},N(295,1,se,E.UM),U.mc=function(t){E.TM(this,t);},l.$9=O(295),Tt.WM=function(t,A,n){Tt.XK.call(this,A,t),this.a=n;},N(140,162,{140:1},Tt.WM),U.a=0,l._9=O(140),Tt._M=function(){Tt._M=D1,Tt.ZM=v(F(l.j7,2),E1,7,0,[v(F(l.j7,1),H,6,15,[-1432180054,-1434146134,-1441453406,-1471067516,-1589095916,-2060839854,1420461386,1454022954,1454024874]),v(F(l.j7,1),H,6,15,[-1431677611,-1431764987,-1432092655,-1433403327,-1438646015,-1459616762,-1543499750,-1879031702,1431634346,1073809066,1073814186,1073834666,1431677610]),v(F(l.j7,1),H,6,15,[-1431737686,-1453057622,-1542187926,-1542188954,-1459351471,-1459351487,Zr,1094713345,1077936134,-1879048186,-1543503866,-1543503846,-1459617766,Kr,Ie,Ie]),v(F(l.j7,1),H,6,15,[-1431737686,-1453057622,-1542187926,-1542188954,-1459351471,-1459351487,Zr,1094713345,1077952518,-1879031802,-1543154682,-1543487462,-1459601382,Kr,Ie,Ie]),v(F(l.j7,1),H,6,15,[mP,mP,-1431741782,-1437330070,-1459351462,-1459351482,Zr,-1858076671,-1874853882,-1879048186,-1543503866,-1543503846,-1459617766,Kr,Ie,Ie]),v(F(l.j7,1),H,6,15,[mP,-1432005270,-1454374890,-1543503871,-1879048191,1073741825,1073741846,1073743210,-1795073366,1079421610,1147841194,-1783977302,kP,kP,-1498764630]),v(F(l.j7,1),H,6,15,[mP,-1432005270,-1454374890,-1543438335,-1878982655,1075138561,1073807382,1073808746,-1795073366,1079421610,1147841194,-1783977302,kP,kP,-1498764630]),v(F(l.j7,1),H,6,15,[1431655786,vA,Te,vA,Te,vA,Te,vA,Te,vA,Te,vA,1145324650]),v(F(l.j7,1),H,6,15,[1431655786,vA,Te,vA,Te,vA,1073758314,1073758250,1074091114,1073758250,1073758314,vA,1145324650]),v(F(l.j7,1),H,6,15,[-1432003926,-1437248086,1767548266,1437248090,1437248086,1433053846,1433037477,-1431672155,-1432004955,1433037477,1433037462,1437248086,1437248090,1767548266,-1437248086,-1432003926]),null]),Tt.XM=v(F(l.j7,1),H,6,15,[1,7,7,5,8,7,8,7,8,7,2,14,2,14,0,0,0,0,8,8,0,0]),Tt.YM=v(F(l.j7,1),H,6,15,[4,20,10,29,18,18,18,18,18,18,16,11,16,11,4,5,4,5,16,16,16,16,0,0,0,0,12,2]),Tt.$M=v(F(l.Ybb,1),st,2,6,["chain.png","eraser.png","hand.png","handPlus.png","fist.png","lasso.png","lassoPlus.png","rect.png","rectPlus.png","zoom.png","invisible.png",null,null,"pointingHand.png"]);},Tt.aN=function(t,A){Tt._M();var n,P,r,i;if(Tt.ZM[A]!=null){for(P=0;P<Tt.ZM[A].length;P++)for(i=Tt.ZM[A][P],r=15;r>=0;r--)(i&3)==0&&C.f3(t,r,P,-1),i>>=2;for(n=0;n<Tt.ZM[A].length;n++)for(i=Tt.ZM[A][n],r=15;r>=0;r--)(i&3)==1&&C.f3(t,r,n,O0),i>>=2;}},Tt.bN=function(t){e._c(),e.ce.call(this,t);},N(105,121,{121:1,105:1},Tt.bN),U.qb=function(t){C.K2(this.B,t.a,t.c,t.b,t.d);},U.rb=function(t){C.I2(this.B,t.a,t.c,t.b,t.d);},U.sb=function(t){C.P2(this.B,new C.n3(t));},U.tb=function(t,A,n){C.H2(this.B,A,n,t);},U.ub=function(t,A,n){C.O2(this.B,t,A,n);},U.vb=function(t){return C.S2(this.B,t).b;},U.wb=function(){return this.b;},U.xb=function(){e.vd(this),this.a=1;},U.Ab=function(t){this.a=t,C.Z2(this.B,this.a);},U.Bb=function(t){C.$2(this.B,t);},U.Cb=function(t){this.b=t,this.B&&C.Y2(this.B,t,false,false);},U.a=0,U.b=0,l.aab=O(105),Tt.nN=function(t,A){y.sj(t.a,A);},Tt.oN=function(t,A){var n,P;for(P=new y.Bqb(t.a);P.a<P.c.a.length;)n=y.Aqb(P),n.mc(A);},Tt.pN=function(){this.a=new y.Mj();},N(221,1,{}),l.bab=O(221),Tt.qN=function(t,A,n,P){Tt.XK.call(this,t,P),this.b=A,this.a=n;},N(156,162,{156:1},Tt.qN),U.a=false,U.b=0,l.dab=O(156),Tt.sN=function(){Tt.sN=D1,Tt.rN=v(F(l.Ybb,1),st,2,6,["none","pressed","released","clicked","entered","exited","moved","dragged"]);},Tt.tN=function(t,A,n,P,r,i,D,u,o,Q){Tt.sN(),Tt.XK.call(this,t,Q),this.b=A,this.c=n,this.g=P,this.i=r,this.f=i,this.d=D,this.a=u,this.e=o;},N(137,162,{137:1},Tt.tN),U.pb=function(){return Tt.rN[this.j]+" x:"+this.g+" y:"+this.i+" button:"+this.b+" clicks:"+this.c+(this.f?" shift":"")+(this.d?" ctrl":"")+(this.a?" alt":"")+" isPopup:"+(this.e?"y":"n");},U.a=false,U.b=0,U.c=0,U.d=false,U.e=false,U.f=false,U.g=0,U.i=0,l.eab=O(137),Tt.vN=function(){},Tt.wN=function(t,A){this.a=t,this.b=A;},N(55,1,{55:1},Tt.vN,Tt.wN),U.pb=function(){return "x:"+this.a+" y:"+this.b;},U.a=0,U.b=0,l.fab=O(55),Tt.yN=function(t,A,n){t.a==t.b.length&&(t.b=y.Iqb(t.b,2*t.a),t.c=y.Iqb(t.c,2*t.a)),t.b[t.a]=A,t.c[t.a]=n,++t.a;},Tt.zN=function(t,A){return t.b[A];},Tt.AN=function(t,A){return t.c[A];},Tt.BN=function(t){t.a>0&&--t.a;},Tt.CN=function(){this.b=j(l.h7,Ft,6,64,15,1),this.c=j(l.h7,Ft,6,64,15,1),this.a=0;},Tt.DN=function(t){this.b=j(l.h7,Ft,6,t,15,1),this.c=j(l.h7,Ft,6,t,15,1),this.a=0;},N(118,1,{404:1},Tt.CN,Tt.DN),U.Hb=function(t,A){var n,P,r;for(r=false,P=this.a-1,n=0;n<this.a;n++)this.c[n]>A!=this.c[P]>A&&t<(this.b[P]-this.b[n])*(A-this.c[n])/(this.c[P]-this.c[n])+this.b[n]&&(r=!r),P=n;return r;},U.a=0,l.gab=O(118),Tt.FN=function(t,A,n){return A>=t.c&&A<=t.c+t.b&&n>=t.d&&n<=t.d+t.a;},Tt.GN=function(t,A){return Tt.FN(t,A.c,A.d)&&Tt.FN(t,A.c+A.b,A.d+A.a);},Tt.HN=function(t,A,n,P,r){t.c=A,t.d=n,t.b=P,t.a=r;},Tt.IN=function(t,A){var n,P,r,i;return r=d.Math.min(t.c,A.c),i=d.Math.min(t.d,A.d),P=d.Math.max(t.c+t.b,A.c+A.b)-r,n=d.Math.max(t.d+t.a,A.d+A.a)-i,new Tt.KN(r,i,P,n);},Tt.JN=function(){},Tt.KN=function(t,A,n,P){this.c=t,this.d=A,this.b=n,this.a=P;},N(35,1,{35:1,404:1},Tt.JN,Tt.KN),U.Hb=function(t,A){return Tt.FN(this,t,A);},U.pb=function(){return "x:"+this.c+" y:"+this.d+" w:"+this.b+" h:"+this.a;},U.a=0,U.b=0,U.c=0,U.d=0,l.hab=O(35),d1.ON=function(t){var A,n,P,r,i;for(n=Lt.Q3(new v1.Lfb(238,238,238)),P=WA&n.a,r=0;r<C.e3(t);r++)for(i=0;i<C.b3(t);i++)A=C.d3(t,r,i),C.f3(t,r,i,(O0&A)+P);},d1.PN=function(){return globalThis.devicePixelRatio||1;},d1.QN=function(t){return Rt(d.Math.round((d1.NN==-1&&(d1.NN=d1.PN()),d1.NN*t)));},d1.NN=-1,d1.TN=function(){d1.TN=D1,d1.SN=v(F(l.j7,1),H,6,15,[5258420,0]),d1.RN=v(F(l.j7,1),H,6,15,[11837695,14737632]);},d1.UN=function(t){d1.TN();var A;A=dP,(T.kjb(w.aAb(A).substr(0,18),xe)||T.kjb(w.aAb(A).substr(0,17),ze)||T.kjb(w.aAb(A).substr(0,14),Ne))&&d1.VN(t,d1.RN);},d1.VN=function(t,A){var n,P,r,i,D;for(i=0;i<C.e3(t);i++)for(D=0;D<C.b3(t);D++)for(n=C.d3(t,i,D),r=n&WA,P=0;P<d1.SN.length;P++)if(r==d1.SN[P]){C.f3(t,i,D,(O0&n)+A[P]);break;}},m.WN=function(t){t=t||{};var A=0;return t.createSymmetryRank&&(A|=1),t.considerStereoheterotopicity&&(A|=2),t.encodeAtomCustomLabels&&(A|=8),t.encodeAtomSelection&&(A|=16),t.assignParitiesToTetrahedralN&&(A|=32),t.coordsAre3d&&(A|=64),t.createPseudoStereoGroups&&(A|=128),t.distinguishRacemicOrGroups&&(A|=256),t.tieBreakFreeValenceAtoms&&(A|=512),t.encodeAtomCustomLabelsWithoutRanking&&(A|=i0),t.neglectAnyStereoInformation&&(A|=j1),A;},m.XN=function(t,A){this.a=new e.Og(t.a,m.WN(A));},N(401,1,{},m.XN),U.getCanMolecule=function(t){var A;return A=e.kg(this.a,t),new m.xT(A);},U.getEncodedCoordinates=function(t){return e.ng(this.a,t);},U.getEncodedMapping=function(){return e.og(this.a);},U.getFinalRank=function(){return this.a.f;},U.getGraphAtoms=function(){return e.pg(this.a);},U.getGraphIndexes=function(){return e.qg(this.a);},U.getIDCode=function(){return e.rg(this.a);},U.getSymmetryRank=function(t){return e.vg(this.a,t);},U.getSymmetryRanks=function(){return this.a.g;},U.hasCIPParityDistinctionProblem=function(){return this.a.d;},U.invalidateCoordinates=function(){this.a.s=null;},U.normalizeEnantiomer=function(){return e.Fg(this.a);},U.setParities=function(){e.Jg(this.a);},l.jab=O(401),m.oO=function(){},m.pO=function(t,A){switch(A){default:case 0:return e.Th(t.a,0);case 1:return e.Th(t.a,1);case 2:return e.Th(t.a,2);case 3:return e.Th(t.a,3);case 4:return e.Th(t.a,4);}},N(405,1,{},m.oO),m.jO=2,m.kO=0,m.lO=1,m.mO=4,m.nO=3,l.iab=O(405),m.uO=function(t){m.f$||m.i$(),this.a=new _.YAb(l.Ueb(t));},N(406,1,{},m.uO),U.getConformerCount=function(){return this.a.p;},U.getNextConformerAsMolecule=function(t){var A,n;return A=t?t.a:null,n=_.MAb(this.a,A),n?n==A?t:new m.xT(n):null;},U.getOneConformerAsMolecule=function(t){var A;return A=_.OAb(this.a,t.a),A?t:null;},U.getPotentialConformerCount=function(){return _.PAb(this.a);},U.pc=function(t,A,n,P){return _.SAb(this.a,t.a,A,n,P);},U.initializeConformers=function(t,A){A===void 0&&(A={});var n=A.strategy;n===void 0&&(n=3);var P=A.maxTorsionSets;P===void 0&&(P=uP);var r=A.use60degreeSteps;return r===void 0&&(r=false),this.pc(t,n,P,r);},m.qO=4,m.rO=3,m.sO=1,m.tO=2,l.kab=O(406),m.BO=function(){},m.CO=function(t,A,n,P,r){return nt.jG(t,A,n,P,r);},N(407,1,{},m.BO),l.lab=O(407),m.EO=function(){throw new Error("drug likeness must be assessed first");},m.FO=function(){m.f$||m.i$(),this.a=new nt.oG();},N(408,1,{},m.FO),U.assessDruglikeness=function(t){var A;return nt.mG(this.a,(A=t.a,!V1.w3&&(V1.w3=new V1.x3()),A));},U.getDetail=function(){var t;return t=this.a.a,!t&&m.EO(),V1.B3(t);},U.getDruglikenessString=function(t){return nt.nG(t.a);},m.DO=-999,l.mab=O(408),m.PO=function(t,A,n){m.f$||m.i$(),m.SO(A),this.a=new K.AC(t.a,A,new y.ztb());},m.SO=function(t){T.kjb(t,Yr)&&!m.MO?(K.uC(),K.BC(Yr,new K.sD("/resources/forcefield/mmff94/outofplane.csv","/resources/forcefield/mmff94/torsion.csv")),m.MO=true):T.kjb(t,Xr)&&!m.NO?(K.uC(),K.BC(Xr,new K.sD(qD,"/resources/forcefield/mmff94/94s/torsion.csv")),m.NO=true):T.kjb(t,sP)&&!m.OO&&(K.uC(),K.BC(sP,new K.sD(qD,"/resources/forcefield/mmff94/94s/torsionPlus.csv")),m.OO=true);},N(409,1,{},m.PO),U._minimise=function(t,A,n){return gA.zB(this.a,t,A);},U.getTotalEnergy=function(){return K.wC(this.a);},U.size=function(){return this.a.b.q;},m.JO=Yr,m.KO=Xr,m.LO=sP,m.MO=false,m.NO=false,m.OO=false,l.nab=O(409),m.rT=function(){m.rT=D1,m.qT=new Tt.JN(),m.yP=v(F(l.Ybb,1),st,2,6,["?","H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr","Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"R4","R5","R6","R7","R8","R9","R10","R11","R12","R13","R14","R15","R16","R1","R2","R3","A","A1","A2","A3",R1,R1,"D","T","X","R","H2","H+","Nnn","HYD","Pol",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"Ala","Arg","Asn","Asp","Cys","Gln","Glu","Gly","His","Ile","Leu","Lys","Met","Phe","Pro","Ser","Thr","Trp","Tyr","Val"]),m.pT=v(F(l.Eeb,1),Yt,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]),m.jR=v(F(l.f7,2),bA,10,0,[null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[0]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[0]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[2,4,6]),v(F(l.f7,1),Y,6,15,[1,3,5,7]),v(F(l.f7,1),Y,6,15,[0]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[2,4,6]),v(F(l.f7,1),Y,6,15,[1,3,5,7]),v(F(l.f7,1),Y,6,15,[0,2]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,v(F(l.f7,1),Y,6,15,[1,2,3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[2,4,6]),v(F(l.f7,1),Y,6,15,[1,3,5,7]),v(F(l.f7,1),Y,6,15,[0,2,4,6]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[2])]),m.CS=v(F(l.f7,2),bA,10,0,[null,v(F(l.f7,1),Y,6,15,[1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),null,null,v(F(l.f7,1),Y,6,15,[-3]),v(F(l.f7,1),Y,6,15,[-2]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),null,v(F(l.f7,1),Y,6,15,[-3]),v(F(l.f7,1),Y,6,15,[-2]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3,4]),v(F(l.f7,1),Y,6,15,[2,3,4,5]),v(F(l.f7,1),Y,6,15,[2,3,6]),v(F(l.f7,1),Y,6,15,[2,3,4,7]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[1,2]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[-3,3,5]),v(F(l.f7,1),Y,6,15,[-2]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[6]),v(F(l.f7,1),Y,6,15,[4,6,7]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[-3,3,5]),v(F(l.f7,1),Y,6,15,[-2,4,6]),v(F(l.f7,1),Y,6,15,[-1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[5]),v(F(l.f7,1),Y,6,15,[6]),v(F(l.f7,1),Y,6,15,[4,6,7]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[1,3]),v(F(l.f7,1),Y,6,15,[1,2]),v(F(l.f7,1),Y,6,15,[1,3]),v(F(l.f7,1),Y,6,15,[2,4]),v(F(l.f7,1),Y,6,15,[3,5]),v(F(l.f7,1),Y,6,15,[-2,2,4]),v(F(l.f7,1),Y,6,15,[-1,1]),null,v(F(l.f7,1),Y,6,15,[1]),v(F(l.f7,1),Y,6,15,[2]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[4]),v(F(l.f7,1),Y,6,15,[4,5]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3,4,5,6]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3,4]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[2,3]),v(F(l.f7,1),Y,6,15,[3])]),m.iP=v(F(l.Ybb,1),st,2,6,[Tr,br,cr]);},m.sT=function(t,A,n,P){return {x:t,y:A,width:n,height:P};},m.tT=function(t){return t.a;},m.uT=function(t,A,n,P){var r;return r=0,A&&(r|=1),n&&(r|=2),P&&(r|=4),e.qr(new e.yr(t.a,r));},m.vT=function(){m.rT(),m.wT.call(this,32,32,null);},m.wT=function(t,A,n){m.rT(),n?this.a=n:this.a=new e.uw(t,A);},m.xT=function(t){m.rT(),m.wT.call(this,0,0,t);},m._T=function(t,A){return {molecule:t,map:A};},m.pU=function(t,A){m.rT();var n;return typeof A===Oe&&(A=true),typeof A===u0?n=m.rU(t,A):typeof A===q0&&(n=m.qU(t,A)),n;},m.qU=function(t,A){return new m.xT(e.Qq(new e._q(false),t,A));},m.rU=function(t,A){return new m.xT(e.Pq(new e._q(A),t));},m.sU=function(t){return m.rT(),new m.xT(e.fs(new e.Cs(),t));},m.tU=function(t){m.rT();var A,n,P;return P=new e.Ds(),n=(P.j=null,e.As(P,new Ct.ogb(new Ct.zgb(t)))?P.j:null),A=P.f==null?e.lp(P.j):P.f,m._T(new m.xT(n),A);},m.uU=function(t,A){m.rT(),A=A||{};var n=!A.noCoordinates,P=!A.noStereo;return m.vU(t,n,P);},m.vU=function(t,A,n){var P,r,i;return i=new e.Iv(),r=new m.vT(),e.yv(i,r.a,w.Czb((P=t,w.xzb(),P)),A,n),r;},m.DU=function(t,A,n,P){return m.rT(),e.no(t,A,n,P);},m.EU=function(t,A){return m.rT(),e.oo(t,A);},m.cV=function(t,A){return m.rT(),e.po(t,A);},m.HV=function(){return m.rT(),e.Rk(),e.Rk(),e.Qk;},m.NW=function(t){return m.rT(),e.qo(t);},m.OW=function(t){return m.rT(),e.ro(t);},m.kY=function(t){m.rT(),e.Rk(),e.Qk=t;},N(42,1,{42:1},m.vT,m.wT,m.xT),U.addAtom=function(t){return e.Uk(this.a,t);},U.addBond=function(t,A){return e.Vk(this.a,t,A);},U.addFragment=function(t,A,n){e.xo(this.a,t.a,A,n);},U.qc=function(){Xt.Lz(this.a);},U.rc=function(t){Xt.Mz(this.a,t);},U.addImplicitHydrogens=function(t){t===void 0?this.qc():this.rc(t);},U.sc=function(){Xt.sz(),Xt.tz(this.a,1);},U.tc=function(t){Xt.tz(this.a,t);},U.addMissingChirality=function(t){t===void 0?this.sc():this.tc(t);},U.addMolecule=function(t){return e.Yk(this.a,t.a);},U.addOrChangeAtom=function(t,A,n,P,r,i,D){return e.$k(this.a,t,A,n,P,r,i,D);},U.addOrChangeBond=function(t,A,n){return e._k(this.a,t,A,n);},U.addRing=function(t,A,n,P,r){return e.al(this.a,t,A,n,P,r);},U.addRingToAtom=function(t,A,n,P){return e.bl(this.a,t,A,n,P);},U.addRingToBond=function(t,A,n,P){return e.cl(this.a,t,A,n,P);},U.addSubstituent=function(t,A){return e.dl(this.a,t.a,A);},U.calculateTorsion=function(t){return e.el(this.a,t);},U.canonizeCharge=function(t){return e.zo(this.a,t,false);},U.changeAtom=function(t,A,n,P,r){return e.fl(this.a,t,A,n,P,r);},U.changeAtomCharge=function(t,A){return e.hl(this.a,t,A);},U.changeBond=function(t,A){return e.il(this.a,t,A);},U.clear=function(){e._v(this.a);},U.convertStereoBondsToSingleBonds=function(t){e.Bo(this.a,t);},U.copyAtom=function(t,A,n,P){return e.ll(this.a,t.a,A,n,P);},U.copyBond=function(t,A,n,P,r,i){return e.nl(this.a,t.a,A,n,P,r,i);},U.copyMolecule=function(t){e.ol(this.a,t.a);},U.copyMoleculeByAtoms=function(t,A,n,P){e.Co(this.a,t.a,A,n,P);},U.copyMoleculeByBonds=function(t,A,n,P){return e.Do(this.a,t.a,A,n,P);},U.copyMoleculeProperties=function(t){e.aw(this.a,t.a);},U.deleteAtom=function(t){e.ql(this.a,t);},U.deleteAtomOrBond=function(t,A){return e.rl(this.a,t,A);},U.deleteAtoms=function(t){return e.sl(this.a,t);},U.deleteBond=function(t){e.tl(this.a,t);},U.deleteBondAndSurrounding=function(t){e.ul(this.a,t);},U.deleteMarkedAtomsAndBonds=function(){return e.vl(this.a);},U.deleteMolecule=function(){e._v(this.a);},U.deleteSelectedAtoms=function(){return e.wl(this.a);},U.ensureHelperArrays=function(t){e.bw(this.a,t);},U.findAlleneCenterAtom=function(t){return e.Fo(this.a,t);},U.findAlleneEndAtom=function(t,A){return e.Go(this.a,t,A);},U.findAtom=function(t,A){return e.xl(this.a,t,A);},U.findBINAPChiralityBond=function(t){return e.Ho(this.a,t);},U.findBond=function(t,A){return e.yl(this.a,t,A);},U.findRingSystem=function(t,A,n,P){e.Io(this.a,t,A,n,P);},U.getAbsoluteAtomParity=function(t){return e.cw(this.a,t);},U.getAbsoluteBondParity=function(t){return e.dw(this.a,t);},U.getAllAtoms=function(){return this.a.q;},U.getAllBonds=function(){return this.a.r;},U.getAllConnAtoms=function(t){return e.Lo(this.a,t);},U.getAllConnAtomsPlusMetalBonds=function(t){return e.Mo(this.a,t);},U.getAllHydrogens=function(t){return e.No(this.a,t);},U.getAromaticRingCount=function(){return e.Oo(this.a);},U.getAtomAbnormalValence=function(t){return e.zl(this.a,t);},U.getAtomCIPParity=function(t){return e.Al(this.a,t);},U.getAtomCharge=function(t){return e.Bl(this.a,t);},U.getAtomColor=function(t){return e.Cl(this.a,t);},U.getAtomCustomLabel=function(t){return e.El(this.a,t);},U.getAtomESRGroup=function(t){return e.Gl(this.a,t);},U.getAtomESRType=function(t){return e.Hl(this.a,t);},U.getAtomLabel=function(t){return e.Il(this.a,t);},U.getAtomList=function(t){return e.Jl(this.a,t);},U.getAtomListString=function(t){return e.Kl(this.a,t);},U.getAtomMapNo=function(t){return e.Ll(this.a,t);},U.getAtomMass=function(t){return e.Ml(this.a,t);},U.getAtomParity=function(t){return e.Nl(this.a,t);},U.getAtomPi=function(t){return e.Qo(this.a,t);},U.getAtomPreferredStereoBond=function(t){return e.Ro(this.a,t);},U.getAtomQueryFeaturesObject=function(t){return V1.u3(this.a,t);},U.getAtomRadical=function(t){return e.Pl(this.a,t);},U.getAtomRingBondCount=function(t){return e.So(this.a,t);},U.getAtomRingCount=function(t,A){return e.To(this.a,t,A);},U.getAtomRingSize=function(t){return e.Uo(this.a,t);},U.getAtomX=function(t){return e.Ql(this.a,t);},U.getAtomY=function(t){return e.Rl(this.a,t);},U.getAtomZ=function(t){return e.Sl(this.a,t);},U.getAtomicNo=function(t){return e.Tl(this.a,t);},U.getAtoms=function(){return this.a.f;},U.getAverageBondLength=function(t){return e.Vo(this.a,t);},U.getAverageTopologicalAtomDistance=function(){return e.Wo(this.a);},U.getBond=function(t,A){return e.Xo(this.a,t,A);},U.getBondAngle=function(t,A){return e.Yl(this.a,t,A);},U.getBondAtom=function(t,A){return e.Zl(this.a,t,A);},U.getBondBridgeMaxSize=function(t){return e.$l(this.a,t);},U.getBondBridgeMinSize=function(t){return e._l(this.a,t);},U.getBondCIPParity=function(t){return e.am(this.a,t);},U.getBondESRGroup=function(t){return e.bm(this.a,t);},U.getBondESRType=function(t){return e.cm(this.a,t);},U.getBondLength=function(t){return e.dm(this.a,t);},U.getBondOrder=function(t){return e.fm(this.a,t);},U.getBondParity=function(t){return e.gm(this.a,t);},U.getBondPreferredStereoBond=function(t){return e.oq(this.a,t);},U.getBondQueryFeatures=function(t){return e.hm(this.a,t);},U.getBondQueryFeaturesObject=function(t){return V1.v3(this.a,t);},U.getBondRingSize=function(t){return e.Yo(this.a,t);},U.getBondType=function(t){return e.im(this.a,t);},U.getBondTypeSimple=function(t){return e.jm(this.a,t);},U.getBonds=function(){return this.a.g;},U.getBounds=function(){var t;return t=e.km(this.a,m.qT),t?m.sT(t.c,t.d,t.b,t.a):null;},U.getCanonizedIDCode=function(t){var A;return A=new e.Og(this.a,t),e.rg(A);},U.getChiralText=function(){return e.ew(this.a);},U.getChirality=function(){return this.a.I;},U.getCompactCopy=function(){return new m.xT(e.fw(this.a));},U.getConnAtom=function(t,A){return e.Zo(this.a,t,A);},U.getConnAtoms=function(t){return e.$o(this.a,t);},U.getConnBond=function(t,A){return e._o(this.a,t,A);},U.getConnBondOrder=function(t,A){return e.ap(this.a,t,A);},U.getDefaultMaxValenceUncharged=function(t){return e.lm(this.a,t);},U.getDiastereotopicAtomIDs=function(){return Xt.uz(this.a);},U.getElectronValenceCorrection=function(t,A){return e.mm(this.a,t,A);},U.getExcludedNeighbourCount=function(t){return e.bp(this.a,t);},U.getExplicitHydrogens=function(t){return e.cp(this.a,t);},U.getFinalRanks=function(t){var A;return A=new e.Og(this.a,t),A.f;},U.getFisherProjectionParity=function(t,A,n,P){return e.ep(this.a,t,A,n,P);},U.getFragmentAtoms=function(t,A){return e.fp(this.a,t,A);},U.getFragmentNumbers=function(t,A,n){return e.hp(this.a,t,A,n);},U.getFragments=function(){var t,A,n;for(t=e.hw(this.a),n=j(l.pab,_r,42,t.length,0,1),A=0;A<t.length;A++)n[A]=new m.xT(t[A]);return n;},U.getFreeValence=function(t){return e.jp(this.a,t);},U.getHandleHydrogenMap=function(){return e.lp(this.a);},U.getHelperArrayStatus=function(){return this.a.T;},U.getHoseCodes=function(t){t=t||{};var A=(typeof t.maxSphereSize===Oe?5:t.maxSphereSize)|0,n=(typeof t.type===Oe?0:t.type)|0;return Xt.wz(this.a,A,n);},U.getIDCode=function(){var t;return t=e.jw(this.a),t==null&&(t=e.rg(new e.Ng(this.a))),t;},U.getIDCodeAndCoordinates=function(){return {idCode:this.getIDCode(),coordinates:this.getIDCoordinates()};},U.getIDCoordinates=function(){var t;return t=e.kw(this.a),t==null&&(t=e.mg(new e.Ng(this.a))),t;},U.getImplicitHigherValence=function(t,A){return e.mp(this.a,t,A);},U.getImplicitHydrogens=function(t){return e.op(this.a,t);},U.getIndex=function(){return V1.C3(this.a);},U.getLowestFreeValence=function(t){return e.pp(this.a,t);},U.getMaxAtoms=function(){return this.a.M;},U.getMaxBonds=function(){return this.a.N;},U.getMaxValence=function(t){return e.om(this.a,t);},U.getMaxValenceUncharged=function(t){return e.pm(this.a,t);},U.getMetalBondedConnAtoms=function(t){return e.rp(this.a,t);},U.getMolecularFormula=function(){return new m.J_(this.a);},U.getMoleculeColor=function(){return this.a.O;},U.getMolweight=function(){return e.sp(this.a);},U.getName=function(){return this.a.P;},U.getNonHydrogenNeighbourCount=function(t){return e.tp(this.a,t);},U.getNumberOfHydrogens=function(){return Xt.Nz(this.a);},U.getOccupiedValence=function(t){return e.vp(this.a,t);},U.getPath=function(t,A,n,P,r){return e.yp(this.a,t,A,n,P,null,r);},U.getPathBonds=function(t,A,n){e.zp(this.a,t,A,n);},U.getPathLength=function(t,A){return e.Ap(this.a,t,A);},U.getRingSet=function(){return new m.o$(e.Ep(this.a));},U.getRotatableBondCount=function(){return e.Fp(this.a);},U.uc=function(t,A,n,P,r,i,D){var u,o,Q,L,s,h,b,c,g,B,k,M;for(Q=true,s=0;s<this.a.q-1;s++)if(e.Ql(this.a,s)!=e.Ql(this.a,s+1)||e.Rl(this.a,s)!=e.Rl(this.a,s+1)){Q=false;break;}return h=Q?e.fw(this.a):this.a,Q&&X.jA(new X.CA(0),h),e.bw(h,95),M=V1.E3(D),L=V1.D3(D),o=new e.Ju(h,L,i),o.J=n,e.be(o,null,new Tt.KN(0,0,t,A),M),u=o.v,e.Td(o,null),k=e.Hu(o),P?(c=Jt(c1(d.Math.round(u.b+r*2))),b=Jt(c1(d.Math.round(u.a+r*2))),g=Jt(c1(d.Math.round(u.c-r))),B=Jt(c1(d.Math.round(u.d-r))),T.xjb(k,'width="\\d+px" height="\\d+px" viewBox="0 0 \\d+ \\d+"','width="'+c+'px" height="'+b+'px" viewBox="'+g+" "+B+" "+c+" "+b+'"')):k;},U.getStereoBond=function(t){return e.Ip(this.a,t);},U.getStereoCenterCount=function(){return e.lw(this.a);},U.vc=function(){return m.tT(this);},U.getStereoProblem=function(t){return e.rm(this.a,t);},U.getSubstituent=function(t,A,n,P,r){return e.Kp(this.a,t,A,n,P.a,r);},U.getSubstituentSize=function(t,A){return e.Lp(this.a,t,A);},U.getSymmetryRank=function(t){return e.mw(this.a,t);},U.getZNeighbour=function(t,A){return e.Mp(this.a,t,A);},U.invalidateHelperArrays=function(t){e.um(this.a,t);},U.inventCoordinates=function(){var t;t=new X.BA(),t.g=new y.Evb(0),X.jA(t,this.a),e.yq(this.a);},U.is3D=function(){return e.vm(this.a);},U.isAlkaliMetal=function(t){return e.Qp(this.a,t);},U.isAllylicAtom=function(t){return e.Rp(this.a,t);},U.isAmideTypeBond=function(t){return e.Sp(this.a,t);},U.isAromaticAtom=function(t){return e.Tp(this.a,t);},U.isAromaticBond=function(t){return e.Up(this.a,t);},U.isAtomConfigurationUnknown=function(t){return e.wm(this.a,t);},U.isAtomMarkedForDeletion=function(t){return e.xm(this.a,t);},U.isAtomParityPseudo=function(t){return e.ym(this.a,t);},U.isAtomStereoCenter=function(t){return e.zm(this.a,t);},U.isAutoMappedAtom=function(t){return e.Am(this.a,t);},U.isBINAPChiralityBond=function(t){return e.Vp(this.a,t);},U.isBondBackgroundHilited=function(t){return e.Bm(this.a,t);},U.isBondBridge=function(t){return e.Cm(this.a,t);},U.isBondForegroundHilited=function(t){return e.Dm(this.a,t);},U.isBondMarkedForDeletion=function(t){return e.Em(this.a,t);},U.isBondParityPseudo=function(t){return e.Fm(this.a,t);},U.isBondParityUnknownOrNone=function(t){return e.Gm(this.a,t);},U.isChalcogene=function(t){return e.Xp(this.a,t);},U.isDelocalizedBond=function(t){return e.Yp(this.a,t);},U.isEarthAlkaliMetal=function(t){return e.Zp(this.a,t);},U.isElectronegative=function(t){return e.Hm(this.a,t);},U.isElectropositive=function(t){return e.Im(this.a,t);},U.isFlatNitrogen=function(t){return e.$p(this.a,t,true);},U.isFragment=function(){return this.a.K;},U.isHalogene=function(t){return e._p(this.a,t);},U.isMarkedAtom=function(t){return e.Km(this.a,t);},U.isMetalAtom=function(t){return e.Lm(this.a,t);},U.isNaturalAbundance=function(t){return e.Mm(this.a,t);},U.isNitrogenFamily=function(t){return e.bq(this.a,t);},U.isOrganicAtom=function(t){return e.Nm(this.a,t);},U.isPseudoRotatableBond=function(t){return e.cq(this.a,t);},U.isPurelyOrganic=function(){return e.Om(this.a);},U.isRingAtom=function(t){return e.fq(this.a,t);},U.isRingBond=function(t){return e.gq(this.a,t);},U.isSelectedAtom=function(t){return e.Pm(this.a,t);},U.isSelectedBond=function(t){return e.Qm(this.a,t);},U.isSimpleHydrogen=function(t){return e.hq(this.a,t);},U.isSmallRingAtom=function(t){return e.iq(this.a,t);},U.isSmallRingBond=function(t){return e.jq(this.a,t);},U.isStabilizedAtom=function(t){return e.kq(this.a,t);},U.isStereoBond=function(t){return e.Rm(this.a,t);},U.markAtomForDeletion=function(t){e.Um(this.a,t);},U.markBondForDeletion=function(t){e.Vm(this.a,t);},U.normalizeAmbiguousBonds=function(){return e.lq(this.a);},U.removeAtomColors=function(){e.Ym(this.a);},U.removeAtomCustomLabels=function(){this.a.t=null;},U.removeAtomMarkers=function(){e.Zm(this.a);},U.removeAtomSelection=function(){e.$m(this.a);},U.removeBondHiliting=function(){e._m(this.a);},U.removeExplicitHydrogens=function(){e.qq(this.a,true);},U.removeQueryFeatures=function(){return e.bn(this.a);},U.renumberESRGroups=function(t){return e.cn(this.a,t);},U.scaleCoords=function(t){e.dn(this.a,t);},U.setAllAtoms=function(t){e.en(this.a,t);},U.setAllBonds=function(t){e.fn(this.a,t);},U.setAssignParitiesToNitrogen=function(t){e.nw(this.a,t);},U.setAtomAbnormalValence=function(t,A){e.gn(this.a,t,A);},U.setAtomCIPParity=function(t,A){e.hn(this.a,t,A);},U.setAtomCharge=function(t,A){e.jn(this.a,t,A);},U.setAtomColor=function(t,A){e.kn(this.a,t,A);},U.setAtomConfigurationUnknown=function(t,A){e.ln(this.a,t,A);},U.setAtomCustomLabel=function(t,A){e.mn(this.a,t,A);},U.setAtomESR=function(t,A,n){e.on(this.a,t,A,n);},U.setAtomList=function(t,A,n){e.qn(this.a,t,A,n);},U.setAtomMapNo=function(t,A,n){e.rn(this.a,t,A,n);},U.setAtomMarker=function(t,A){e.sn(this.a,t,A);},U.setAtomMass=function(t,A){e.tn(this.a,t,A);},U.setAtomParity=function(t,A,n){e.un(this.a,t,A,n);},U.setAtomQueryFeature=function(t,A,n){e.vn(this.a,t,A,n);},U.setAtomRadical=function(t,A){e.wn(this.a,t,A);},U.setAtomSelection=function(t,A){e.xn(this.a,t,A);},U.setAtomX=function(t,A){e.zn(this.a,t,A);},U.setAtomY=function(t,A){e.An(this.a,t,A);},U.setAtomZ=function(t,A){e.Bn(this.a,t,A);},U.setAtomicNo=function(t,A){e.Cn(this.a,t,A);},U.setBondAtom=function(t,A,n){e.Dn(this.a,t,A,n);},U.setBondBackgroundHiliting=function(t,A){e.En(this.a,t,A);},U.setBondCIPParity=function(t,A){e.Fn(this.a,t,A);},U.setBondESR=function(t,A,n){e.Gn(this.a,t,A,n);},U.setBondForegroundHiliting=function(t,A){e.Hn(this.a,t,A);},U.setBondOrder=function(t,A){e.In(this.a,t,A);},U.setBondParity=function(t,A,n){e.Jn(this.a,t,A,n);},U.setBondParityUnknownOrNone=function(t){e.Kn(this.a,t);},U.setBondQueryFeature=function(t,A,n){e.Ln(this.a,t,A,n);},U.setBondType=function(t,A){e.Mn(this.a,t,A);},U.setChirality=function(t){e.Nn(this.a,t);},U.setFragment=function(t){e.On(this.a,t);},U.setHelperValidity=function(t){e.Pn(this.a,t);},U.setHydrogenProtection=function(t){e.Qn(this.a,t);},U.setMaxAtoms=function(t){e.Rn(this.a,t);},U.setMaxBonds=function(t){e.Sn(this.a,t);},U.setMoleculeColor=function(t){e.Tn(this.a,t);},U.setName=function(t){e.Un(this.a,t);},U.setParitiesValid=function(t){e.vq(this.a,t);},U.setStereoBondFromAtomParity=function(t){e.wq(this.a,t);},U.setStereoBondFromBondParity=function(t){e.xq(this.a,t);},U.setStereoBondsFromParity=function(){e.yq(this.a);},U.setToRacemate=function(){this.a.L=true;},U.setUnknownParitiesToExplicitlyUnknown=function(){e.ow(this.a);},U.shareSameFragment=function(t,A){return e.Ap(this.a,t,A)!=-