/*
 * mzdata v4.0.0
 * Read and explore mzData v1.05 files
 * https://github.com/cheminfo-js/mzData#readme
 *
 * Licensed under the MIT license.
 */
function parseString(value) {
  if (value.length === 4 || value.length === 5) {
    const lowercase = value.toLowerCase();
    if (lowercase === 'true') return true;
    if (lowercase === 'false') return false;
  }
  const number = Number(value);
  if (number === 0 && !value.includes('0')) {
    return value;
  }
  if (!Number.isNaN(number)) return number;
  return value;
}

const utf8Decoder$1 = new TextDecoder();
const decoder$5 = {
  decode: array => {
    return utf8Decoder$1.decode(array);
  }
};
const defaultOptions = {
  trimValues: true,
  attributesNodeName: '',
  ignoreAttributes: false,
  ignoreNameSpace: false,
  allowBooleanAttributes: false,
  parseAttributesString: true,
  textNodeName: '#text',
  arrayMode: false,
  cdataTagName: false,
  tagNameProcessor: name => name,
  attributeNameProcessor: name => `$${name}`,
  tagValueProcessor: value => {
    const string = decoder$5.decode(value).replaceAll('\r', '');
    return parseString(string);
  },
  attributeValueProcessor: value => parseString(value),
  stopNodes: []
};

class XMLNode {
  tagName;
  parent;
  children;
  attributes;
  bytes;
  startIndex;
  tagValueProcessor;
  cachedValue;
  constructor(tagName, parent, bytes, tagValueProcessor) {
    this.tagName = tagName;
    this.parent = parent;
    this.children = Object.create(null); //child tags
    this.attributes = Object.create(null); //attributes map
    this.bytes = bytes; //text only
    this.tagValueProcessor = tagValueProcessor;
    this.startIndex = -1;
  }
  append(toAppend) {
    if (this.bytes.length === 0) {
      this.bytes = toAppend;
      return;
    }
    const arrayConcat = new Uint8Array(this.bytes.length + toAppend.length);
    arrayConcat.set(this.bytes);
    arrayConcat.set(toAppend, this.bytes.length);
    this.bytes = arrayConcat;
  }
  get value() {
    if (this.cachedValue === undefined) {
      const value = this.tagValueProcessor(this.bytes, this);
      this.cachedValue = value;
    }
    return this.cachedValue;
  }
  addChild(child) {
    if (Array.isArray(this.children[child.tagName])) {
      //already presents
      this.children[child.tagName].push(child);
    } else {
      this.children[child.tagName] = [child];
    }
  }
}

function arrayIndexOf(array, referenceArray, index = 0) {
  let found = 0;
  let foundIndex = -1;
  for (let i = index; i < array.length && found < referenceArray.length; i++) {
    if (array[i] === referenceArray[found]) {
      if (!found) {
        foundIndex = i;
      }
      found++;
    } else if (found > 0) {
      let j = 0;
      for (; j <= found && array[foundIndex + j] === array[foundIndex + found]; j++);
      if (j < found + 1) {
        foundIndex = -1;
        found = 0;
      } else {
        foundIndex++;
      }
    } else {
      found = 0;
      foundIndex = -1;
    }
  }
  if (found !== referenceArray.length) {
    foundIndex = -1;
  }
  return foundIndex;
}

/* eslint-disable @typescript-eslint/no-unused-vars */
function arrayTrim(array, arg) {
  let i = 0;
  let j = array.length - 1;
  for (; i < array.length && array[i] <= 0x20; i++);
  for (; j >= i && array[j] <= 0x20; j--);
  if (i === 0 && j === array.length - 1) return array;
  return array.subarray(i, j + 1);
}

const utf8Decoder = new TextDecoder();
const decoder$4 = {
  decode: array => {
    return utf8Decoder.decode(array);
  }
};

/**
 * Search for the corresponding closing tag '>'
 * @param data
 * @param i
 * @returns
 */
function closingIndexForOpeningTag(data, i) {
  let attrBoundary;
  let endIndex = 0;
  for (let index = i; index < data.length; index++) {
    let byte = data[index];
    if (attrBoundary) {
      if (byte === attrBoundary) attrBoundary = 0; //reset
    } else if (byte === 0x22 || byte === 0x27) {
      attrBoundary = byte;
    } else if (byte === 0x3e) {
      return {
        data: decoder$4.decode(data.subarray(i, i + endIndex)),
        index
      };
    } else if (byte === 0x09) {
      byte = 0x20;
    }
    endIndex++;
  }
  throw new Error('Could not find closing tag');
}

function findClosingIndex(xmlData, str, i, errMsg) {
  const closingIndex = arrayIndexOf(xmlData, str, i);
  if (closingIndex === -1) {
    throw new Error(errMsg);
  } else {
    return closingIndex + str.length - 1;
  }
}

function getAllMatches(string, regex) {
  return Array.from(string.matchAll(regex));
}
function isEmptySimpleObject(object) {
  // fastest implementation: https://jsbench.me/qfkqv692c8/1
  // eslint-disable-next-line no-unreachable-loop
  for (const key in object) {
    return false;
  }
  return true;
}
function isEmptyObject(object) {
  // fastest implementation: https://jsbench.me/qfkqv692c8/1
  // eslint-disable-next-line no-unreachable-loop
  for (const key in object) {
    return false;
  }
  return true;
}
/**
 * Copy all the properties of a into b.
 * @param target
 * @param source
 * @param arrayMode
 */
function merge(target, source, arrayMode) {
  if (!source) return;
  for (const key in source) {
    if (arrayMode === 'strict') {
      target[key] = [source[key]];
    } else {
      target[key] = source[key];
    }
  }
}
/**
 * Check if a tag name should be treated as array
 * @param tagName - the node tagName
 * @param arrayMode - the array mode option
 * @param parentTagName - the parent tag name
 * @returns true if node should be parsed as array
 */
function isTagNameInArrayMode(tagName, arrayMode, parentTagName) {
  if (arrayMode === false) {
    return false;
  } else if (arrayMode instanceof RegExp) {
    return arrayMode.test(tagName);
  } else if (typeof arrayMode === 'function') {
    return arrayMode(tagName, parentTagName);
  }
  return arrayMode === 'strict';
}

const newLocal = String.raw`([^\s=]+)\s*(=\s*(['"])(.*?)\3)?`;
const attrsRegx = new RegExp(newLocal, 'g');
//Attributes are strings so no point in using arrayBuffers here
function parseAttributesString(string, options) {
  const {
    ignoreAttributes
  } = options;
  if (ignoreAttributes) {
    return;
  }
  string = string.replaceAll(/\r?\n/g, ' ');
  const matches = getAllMatches(string, attrsRegx);
  // argument 1 is the key, argument 4 is the value
  const attributes = {};
  for (const match of matches) {
    const attributeName = resolveNameSpace(match[1], options);
    if (attributeName.length > 0) {
      if (match[4] !== undefined) {
        if (options.trimValues) {
          match[4] = match[4].trim();
        }
        if (options.attributeValueProcessor) {
          attributes[attributeName] = options.attributeValueProcessor(match[4], attributeName);
        }
      } else if (options.allowBooleanAttributes) {
        attributes[attributeName] = true;
      }
    }
  }
  if (isEmptySimpleObject(attributes)) return;
  return attributes;
}
function resolveNameSpace(tagName, options) {
  if (options.ignoreNameSpace) {
    const tags = tagName.split(':');
    const prefix = tagName.startsWith('/') ? '/' : '';
    if (tags[0] === 'xmlns') {
      return '';
    }
    if (tags.length === 2) {
      tagName = prefix + tags[1];
    }
  }
  return tagName;
}

function removeNameSpaceIfNeeded(tagName, options) {
  if (!options.ignoreNameSpace) {
    return tagName;
  }
  const colonIndex = tagName.indexOf(':');
  if (colonIndex !== -1) {
    tagName = tagName.slice(colonIndex + 1);
  }
  return tagName;
}

function getTraversable(xmlData, options) {
  const {
    tagValueProcessor
  } = options;
  const traversable = new XMLNode('!xml', undefined, new Uint8Array(0), tagValueProcessor);
  let currentNode = traversable;
  let dataSize = 0;
  let dataIndex = 0;
  for (let i = 0; i < xmlData.length; i++) {
    if (xmlData[i] === 0x3c) {
      // <
      const xmlData1 = xmlData[i + 1];
      const xmlData2 = xmlData[i + 2];
      if (xmlData1 === 0x2f) {
        // </ Closing Tag
        const closeIndex = findClosingIndex(xmlData, [0x3e],
        //>
        i, 'Closing Tag is not closed.');
        let tagName = decoder$4.decode(arrayTrim(xmlData.subarray(i + 2, closeIndex)));
        tagName = removeNameSpaceIfNeeded(tagName, options);
        if (currentNode) {
          currentNode.append(options.trimValues ? arrayTrim(xmlData.subarray(dataIndex, dataIndex + dataSize)) : xmlData.subarray(dataIndex, dataIndex + dataSize));
        }
        if (options.stopNodes?.length && options.stopNodes.includes(currentNode.tagName)) {
          currentNode.children = {};
          if (currentNode.attributes === undefined) {
            currentNode.attributes = {};
          }
          currentNode.bytes = xmlData.subarray(currentNode.startIndex + 1, i);
        }
        currentNode = currentNode.parent;
        i = closeIndex;
        dataSize = 0;
        dataIndex = i + 1;
      } else if (xmlData1 === 0x3f) {
        // <? PI, processing instruction
        i = findClosingIndex(xmlData, [0x3f, 0x3e], i, 'Pi Tag is not closed.');
      } else if (
      //!-- comment
      xmlData1 === 0x21 && xmlData2 === 0x2d && xmlData[i + 3] === 0x2d) {
        i = findClosingIndex(xmlData, [0x2d, 0x2d, 0x3e],
        //-->
        i, 'Comment is not closed.');
        if (currentNode && dataSize !== 0 && currentNode.tagName !== '!xml') {
          currentNode.append(options.trimValues ? arrayTrim(xmlData.subarray(dataIndex, dataSize + dataIndex)) : xmlData.subarray(dataIndex, dataSize + dataIndex));
        }
        dataSize = 0;
        dataIndex = i + 1;
        //!D
      } else if (xmlData1 === 0x21 && xmlData2 === 0x44) {
        // <!D
        const closeIndex = findClosingIndex(xmlData, [0x3e],
        //>
        i, 'DOCTYPE is not closed.');
        const tagExp = xmlData.subarray(i, closeIndex);
        if (arrayIndexOf(tagExp, [0x5b]) >= 0) {
          i = arrayIndexOf(xmlData, [0x5d, 0x3e], i) + 1;
        } else {
          i = closeIndex;
        } //![
      } else if (xmlData1 === 0x21 && xmlData2 === 0x5b) {
        // <![CDATA[some stuff]]>
        const closeIndex = findClosingIndex(xmlData, [0x5d, 0x5d, 0x3e],
        //]]>
        i, 'CDATA is not closed.') - 2;
        const tagExp = xmlData.subarray(i + 9, closeIndex);
        //considerations
        //1. CDATA will always have parent node
        //2. A tag with CDATA is not a leaf node so it's value would be string type.
        if (dataSize !== 0) {
          const value = options.trimValues ? arrayTrim(xmlData.subarray(dataIndex, dataIndex + dataSize)) : xmlData.subarray(dataIndex, dataIndex + dataSize);
          currentNode.append(value);
        }
        if (options.cdataTagName) {
          //add cdata node
          const childNode = new XMLNode(options.cdataTagName, currentNode, tagExp, tagValueProcessor);
          currentNode.addChild(childNode);
          //add rest value to parent node
          if (tagExp) {
            childNode.bytes = tagExp;
          }
        } else {
          currentNode.append(tagExp);
        }
        i = closeIndex + 2;
        dataSize = 0;
        dataIndex = i + 1;
      } else {
        //Opening a normal tag
        const parsedOpeningTag = closingIndexForOpeningTag(xmlData, i + 1);
        const tagData = parsedOpeningTag.data.replaceAll(/\r?\n|\t/g, ' ');
        const closeIndex = parsedOpeningTag.index;
        const separatorIndex = tagData.indexOf(' ');
        let shouldBuildAttributesMap = true;
        let tagName = separatorIndex !== -1 ? tagData.slice(0, Math.max(0, separatorIndex)).replace(/\s+$/, '') : tagData;
        let tagAttributes = separatorIndex !== -1 ? tagData.slice(separatorIndex + 1) : '';
        if (options.ignoreNameSpace) {
          const colonIndex = tagName.indexOf(':');
          if (colonIndex !== -1) {
            tagName = tagName.slice(colonIndex + 1);
            shouldBuildAttributesMap = tagName !== parsedOpeningTag.data.slice(colonIndex + 1);
          }
        }
        //save text to parent node
        if (currentNode && dataSize !== 0 && currentNode.tagName !== '!xml') {
          currentNode.append(options.trimValues ? arrayTrim(xmlData.subarray(dataIndex, dataIndex + dataSize)) : xmlData.subarray(dataIndex, dataIndex + dataSize));
        }
        if (tagData.length > 0 && tagData.endsWith('/')) {
          //selfClosing tag
          if (tagAttributes) {
            // <abc def="123"/>
            tagAttributes = tagAttributes.slice(0, Math.max(0, tagAttributes.length - 1));
          } else {
            // <abc/>
            tagName = tagName.slice(0, Math.max(0, tagName.length - 1));
          }
          const childNode = new XMLNode(tagName, currentNode, new Uint8Array(0), tagValueProcessor);
          if (tagAttributes) {
            childNode.attributes = parseAttributesString(tagAttributes, options);
          }
          currentNode.addChild(childNode);
        } else {
          //opening tag
          const childNode = new XMLNode(tagName, currentNode, new Uint8Array(0), tagValueProcessor);
          if (options.stopNodes?.length && options.stopNodes.includes(childNode.tagName)) {
            childNode.startIndex = closeIndex;
          }
          if (tagAttributes && shouldBuildAttributesMap) {
            childNode.attributes = parseAttributesString(tagAttributes, options);
          }
          currentNode.addChild(childNode);
          currentNode = childNode;
        }
        i = closeIndex;
        dataSize = 0;
        dataIndex = i + 1;
      }
    } else {
      dataSize++;
    }
  }
  return traversable;
}

/**
 *
 * @param node
 * @param options
 * @param parentTagName
 * @returns
 */
function traversableToJSON(node, options, parentTagName) {
  const {
    arrayMode,
    tagNameProcessor,
    attributeNameProcessor,
    textNodeName
  } = options;
  const result = {};
  // when no child node or attr is present
  if ((!node.children || isEmptyObject(node.children)) && (!node.attributes || isEmptySimpleObject(node.attributes))) {
    return node.value;
  }
  // otherwise create a textnode if node has some text
  if (node.bytes.length > 0) {
    const asArray = isTagNameInArrayMode(node.tagName, arrayMode, parentTagName);
    result[textNodeName] = asArray ? [node.value] : node.value;
  }
  if (node.attributes && !isEmptySimpleObject(node.attributes)) {
    let attributes = options.parseAttributesString ? {} : node.attributes;
    if (attributeNameProcessor) {
      // need to rename the attributes
      const renamedAttributes = {};
      for (const attributeName in node.attributes) {
        const newAttributeName = attributeNameProcessor(attributeName);
        renamedAttributes[newAttributeName] = node.attributes[attributeName];
      }
      attributes = renamedAttributes;
    }
    if (options.attributesNodeName) {
      const encapsulatedAttributes = {
        [options.attributesNodeName]: attributes
      };
      attributes = encapsulatedAttributes;
    }
    merge(result, attributes, arrayMode);
  }
  for (const tagName in node.children) {
    const nodes = node.children[tagName];
    const newTagName = tagNameProcessor ? tagNameProcessor(tagName, nodes) : tagName;
    if (nodes?.length > 1) {
      result[tagName] = [];
      for (const child of nodes) {
        result[newTagName].push(traversableToJSON(child, options, tagName));
      }
    } else {
      const subResult = traversableToJSON(nodes[0], options, tagName);
      const asArray = arrayMode === true && typeof subResult === 'object' || isTagNameInArrayMode(tagName, arrayMode, parentTagName);
      result[newTagName] = asArray ? [subResult] : subResult;
    }
  }
  return result;
}

/**
 * Parse an ArrayBuffer or Uint8Array representing an XML and return an object
 * @param xmlData
 * @param options
 */
function parse(xmlData, options = {}) {
  if (typeof xmlData === 'string') {
    const encoder = new TextEncoder();
    xmlData = encoder.encode(xmlData);
  }
  if (!ArrayBuffer.isView(xmlData)) {
    xmlData = new Uint8Array(xmlData);
  }
  const realOptions = {
    ...defaultOptions,
    ...options
  };
  const traversable = getTraversable(xmlData, realOptions);
  return traversableToJSON(traversable, realOptions);
}

/**
 * Resolves all promises in an object recursively. The promise with be replaced by the resolved value.
 * The changes are therefore done in-place !
 * @param object
 * @returns
 */
async function recursiveResolve(object) {
  if (typeof object !== 'object') return object;
  const promises = [];
  await appendPromises(object, promises);
  await Promise.all(promises);
  return object;
}
function appendPromises(object, promises) {
  if (typeof object !== 'object') return object;
  for (const key in object) {
    if (typeof object[key].then === 'function') {
      promises.push(object[key].then(value => object[key] = value));
    } else if (typeof object[key] === 'object') {
      appendPromises(object[key], promises);
    }
  }
  return object;
}

const base64codes$1 = Uint8Array.from([255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 62, 255, 255, 255, 63, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 255, 255, 255, 0, 255, 255, 255, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 255, 255, 255, 255, 255, 255, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51]);
/**
 * Convert a Uint8Array containing a base64 encoded bytes to a Uint8Array containing decoded values
 * @param input
 * @returns a Uint8Array containing the decoded bytes
 */
function decode(input) {
  if (!ArrayBuffer.isView(input)) {
    input = new Uint8Array(input);
  }
  if (input.length % 4 !== 0) {
    throw new Error('Unable to parse base64 string.');
  }
  const output = new Uint8Array(3 * (input.length / 4));
  if (input.length === 0) return output;
  const missingOctets = input.at(-2) === 61 ? 2 : input.at(-1) === 61 ? 1 : 0;
  for (let i = 0, j = 0; i < input.length; i += 4, j += 3) {
    const buffer = base64codes$1[input[i]] << 18 | base64codes$1[input[i + 1]] << 12 | base64codes$1[input[i + 2]] << 6 | base64codes$1[input[i + 3]];
    output[j] = buffer >> 16;
    output[j + 1] = buffer >> 8 & 0xff;
    output[j + 2] = buffer & 0xff;
  }
  return output.subarray(0, output.length - missingOctets);
}

const base64codes = Uint8Array.from([65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 43, 47]);

/*
3 bytes are encoded in 4 bytes of base64
11111122 22223333 33444444
We want to be the fastest possible, so we will use a lookup table to convert 12 bits to 2 bytes of base64
But in order still to avoid one operation we will create 2 of those lookup tables.
- One for 2222 11111122
- One for 3333 33444444
*/
// 2222 11111122
const base64codes1 = new Uint32Array(64 * 64);
for (let i = 0; i < 64; i++) {
  for (let j = 0; j < 64; j++) {
    const index = i << 2 | (j & 0x30) >> 4 | (j & 0x0f) << 8;
    base64codes1[index] = base64codes[i] | base64codes[j] << 8;
  }
}
// 3333 33444444 that we store on the bits 16->31 just to allow to make directly the OR with the previous value
const base64codes2 = new Uint32Array(64 * 64);
for (let i = 0; i < 64; i++) {
  for (let j = 0; j < 64; j++) {
    const index = i << 6 | j;
    base64codes2[index] = base64codes[i] << 16 | base64codes[j] << 24;
  }
}

async function inflate(zlibCompressedData) {
  const inputStream = new ReadableStream({
    start(controller) {
      controller.enqueue(zlibCompressedData);
      controller.close();
    }
  });
  const decompressedStream = inputStream.pipeThrough(new DecompressionStream('deflate'));
  const reader = decompressedStream.getReader();
  const chunks = [];
  let totalLength = 0;
  while (true) {
    // eslint-disable-next-line no-await-in-loop
    const {
      value,
      done
    } = await reader.read();
    if (done) break;
    chunks.push(value);
    totalLength += value.length;
  }

  // Combine chunks into a single Uint8Array
  const decompressedData = new Uint8Array(totalLength);
  let offset = 0;
  for (const chunk of chunks) {
    decompressedData.set(chunk, offset);
    offset += chunk.length;
  }
  return decompressedData;
}

async function decodeData(data, options = {}) {
  if (!(data instanceof Uint8Array)) {
    throw new TypeError('data should be an Uint8Array');
  }
  let {
    endian = 'little',
    precision,
    float = true,
    compression = '',
    base64 = true,
    ontologies = {}
  } = options;
  if ('MS:1000519' in ontologies) {
    precision = 32;
    float = false;
  }
  if ('MS:1000520' in ontologies) precision = 16;
  if ('MS:1000521' in ontologies) precision = 32;
  if ('MS:1000522' in ontologies) {
    float = false;
    precision = 64;
  }
  if ('MS:1000523' in ontologies) precision = 64;
  if ('MS:1000574' in ontologies) compression = 'zlib';
  if (base64) {
    data = decode(data);
  }
  switch (compression.toLowerCase()) {
    case 'zlib':
      data = await inflate(data);
      break;
    case '':
    case 'none':
      break;
    default:
      throw new Error(`Unknown compression algorithm: ${compression}`);
  }
  switch (endian.toLowerCase()) {
    case 'little':
      break;
    case 'network':
    case 'big':
      {
        // we will invert in place the data
        let step;
        switch (precision) {
          case 32:
            step = 4;
            break;
          case 64:
            step = 8;
            break;
          default:
            throw new Error('Can not process bigendian file');
        }
        for (let i = 0; i < data.length - data.length % step; i += step) {
          for (let j = 0; j < step / 2; j++) {
            const temp = data[i + j];
            data[i + j] = data[i + step - 1 - j];
            data[i + step - 1 - j] = temp;
          }
        }
      }
      break;
    default:
      throw new TypeError(`Attributes endian not correct: ${endian}`);
  }

  /*
       We should take care that the length of the Uint8Array is correct but the buffer
       may be a little bit bigger because when decoding base 64 it may end with = or ==
       and we plan the size in the buffer.
      */

  if (float) {
    switch (precision) {
      case 32:
        return new Float32Array(data.buffer, data.byteOffset, data.byteLength / Float32Array.BYTES_PER_ELEMENT);
      case 64:
        return new Float64Array(data.buffer, data.byteOffset, data.byteLength / Float64Array.BYTES_PER_ELEMENT);
      default:
        throw new TypeError(`Incorrect precision: ${precision}`);
    }
  } else {
    switch (precision) {
      case 32:
        return new Int32Array(data.buffer, data.byteOffset, data.byteLength / Int32Array.BYTES_PER_ELEMENT);
      case 64:
        return new BigInt64Array(data.buffer, data.byteOffset, data.byteLength / BigInt64Array.BYTES_PER_ELEMENT);
      default:
        throw new TypeError(`Incorrect precision: ${precision}`);
    }
  }
}

function parseCvParam$1(cvParam) {
  let result = {};
  if (!cvParam) return result;
  let cvParams;
  if (Array.isArray(cvParam)) {
    cvParams = cvParam;
  } else {
    cvParams = [cvParam];
  }
  for (let param of cvParams) {
    let attributes = param.attributes;
    if (attributes.name) {
      result[attributes.name.toLowerCase()] = {
        accession: attributes.accession,
        cvLabel: attributes.cvLabel,
        value: attributes.value,
        name: attributes.name
      };
    }
  }
  return result;
}

function processMetadata(parsed, metadata) {
  if (!parsed || !parsed.description) return;
  let description = parsed.description;
  if (description.dataProcessing) {
    let dataProcessing = description.dataProcessing;
    if (dataProcessing.software && dataProcessing.software.name) {
      metadata.software = dataProcessing.software.name;
    }
  }
  if (description.instrument) {
    let instrument = description.instrument;
    if (instrument.analyzerList && instrument.analyzerList.analyzer) {
      let analyzer = instrument.analyzerList.analyzer;
      let cvParam = parseCvParam$1(analyzer.cvParam);
      if (cvParam.analyzertype) {
        metadata.analyzer = cvParam.analyzertype.value;
      }
    }
    if (instrument.detector) {
      let detector = instrument.detector;
      let cvParam = parseCvParam$1(detector.cvParam);
      if (cvParam.detectortype) {
        metadata.detector = cvParam.detectortype.value;
      }
    }
  }
}

function processSpectrumList$2(parsed, times, msData) {
  if (!parsed || !parsed.spectrumList || !parsed.spectrumList.spectrum) return;
  let spectrumList = parsed.spectrumList.spectrum;
  for (let spectrum of spectrumList) {
    let info = parseCvParam$1(spectrum.spectrumDesc.spectrumSettings.spectrumInstrument.cvParam);
    times.push(info.timeinminutes.value);
    let mzArray = spectrum.mzArrayBinary.data['#text'] || [];
    let intensity = spectrum.intenArrayBinary.data['#text'] || [];
    msData.push([mzArray, intensity]);
  }
}

const decoder$3 = new TextDecoder();

/**
 *
 * @param {*} arrayBuffer
 * @param {import('../Options.js').Options} [options={}]
 * @returns
 */
async function parseMzData(arrayBuffer, options = {}) {
  const {
    logger = console
  } = options;
  const result = {
    metadata: {},
    times: [],
    series: {
      ms: {
        data: []
      }
    }
  };
  let parsed = parse(arrayBuffer, {
    attributesNodeName: 'attributes',
    attributeNameProcessor: attributeName => attributeName,
    tagValueProcessor: (value, node) => {
      if (node.tagName !== 'data') return decoder$3.decode(value);
      const promise = decodeData(node.bytes, node.attributes);
      // avoid unhandled promise rejection and swallow the error
      promise.catch(error => {
        logger.error('error decoding base64', error);
        return [];
      });
      return promise;
    }
  });
  await recursiveResolve(parsed);
  processMetadata(parsed.mzData, result.metadata);
  processSpectrumList$2(parsed.mzData, result.times, result.series.ms.data);
  return result;
}

function parseCvParam(cvParam) {
  let result = {};
  if (!cvParam) return result;
  let cvParams;
  if (Array.isArray(cvParam)) {
    cvParams = cvParam;
  } else {
    cvParams = [cvParam];
  }
  for (let parameter of cvParams) {
    let attributes = parameter.attributes;
    if (attributes.name) {
      result[attributes.accession] = attributes;
    }
  }
  return result;
}

function processSpectrumList$1(parsed, times, msData) {
  if (!parsed || !parsed.run || !parsed.run.spectrumList || !parsed.run.spectrumList.spectrum) {
    return;
  }
  let spectrumList = parsed.run.spectrumList.spectrum;
  for (let spectrum of spectrumList) {
    if (!spectrum.binaryDataArrayList) continue;
    let scanList = spectrum.scanList;
    if (Array.isArray(scanList)) throw new Error('Unsupported scanList');
    let scan = scanList.scan;
    if (typeof scan !== 'object') continue;
    if (Array.isArray(scan)) {
      throw new Error('processSpectrumList: scan may not be an array');
    }
    const cvParam = parseCvParam(scan.cvParam);
    times.push(cvParam['MS:1000016'].value);
    const dataArrayList = spectrum.binaryDataArrayList.binaryDataArray;
    if (dataArrayList.length !== 2) {
      throw new Error('Can not decodeData because length !== 2');
    }
    const first = dataArrayList[0];
    const firstCVParams = parseCvParam(first.cvParam);
    const second = dataArrayList[1];
    const secondCVParams = parseCvParam(second.cvParam);

    // MS:1000514 - m/z array
    // MS:1000515 - intensity array
    if (firstCVParams['MS:1000514'] && secondCVParams['MS:1000515']) {
      msData.push([first.binary, second.binary]);
    }
    if (firstCVParams['MS:1000515'] && secondCVParams['MS:1000514']) {
      msData.push([second.binary, first.binary]);
    }
  }
}

const decoder$2 = new TextDecoder();

// https://www.psidev.info/mzml
// CV = Controlled vocabulary
async function parseMzML(mzmlBuffer, options = {}) {
  const {
    logger = console,
    rawData
  } = options;
  if (!(mzmlBuffer instanceof ArrayBuffer)) {
    throw new TypeError('mzmlBuffer should be an ArrayBuffer');
  }
  if (rawData && !(rawData instanceof ArrayBuffer)) {
    throw new TypeError('rawData should be an ArrayBuffer');
  }
  const rawDataUint8Array = rawData && new Uint8Array(rawData);
  const result = {
    metadata: {},
    times: [],
    series: {
      ms: {
        data: []
      }
    }
  };
  const referenceableParamGroups = {};
  let parsed = parse(mzmlBuffer, {
    attributesNodeName: 'attributes',
    attributeNameProcessor: attributeName => attributeName,
    tagNameProcessor: (name, nodes) => {
      switch (name) {
        case 'referenceableParamGroupList':
          {
            const children = nodes[0]?.children?.referenceableParamGroup;
            for (const group of children) {
              const id = group.attributes?.id;
              referenceableParamGroups[id] = group.children;
            }
          }
          break;
        case 'referenceableParamGroupRef':
          for (const node of nodes) {
            // need to append the references children to the parent
            const ref = node.attributes.ref;
            if (referenceableParamGroups[ref]) {
              const parent = node.parent;
              parent.children = parent.children || {};
              for (const key in referenceableParamGroups[ref]) {
                parent.children[key] = parent.children[key] || [];
                parent.children[key].push(...referenceableParamGroups[ref][key]);
              }
            }
          }
          break;
      }
      return name;
    },
    tagValueProcessor: (value, node) => {
      if (node.tagName !== 'binary') return decoder$2.decode(value);
      const referenceableParamGroupRefs = node.parent?.children?.referenceableParamGroupRef?.map(ref => ref.attributes.ref) || [];
      const ontologies = {};
      referenceableParamGroupRefs.forEach(ref => {
        if (referenceableParamGroups[ref]) {
          Object.assign(ontologies, referenceableParamGroups[ref]);
        }
      });
      node.parent.children.cvParam.forEach(cv => {
        ontologies[cv.attributes.accession] = cv.attributes.value;
      });
      let promise;
      if ('IMS:1000102' in ontologies) {
        // external data offset
        const offset = parseInt(ontologies['IMS:1000102'], 10);
        const encodedLength = parseInt(ontologies['IMS:1000104'], 10);
        promise = decodeData(rawDataUint8Array.subarray(offset, offset + encodedLength), {
          ontologies,
          base64: false
        });
      } else {
        promise = decodeData(node.bytes, {
          ontologies,
          base64: true
        });
      }

      // avoid unhandled promise rejection and swallow the error
      promise.catch(error => {
        logger.error('error decoding base64', error);
        return [];
      });
      return promise;
    }
  });
  // parsed file still contains promises
  await recursiveResolve(parsed);
  const mzML = parsed.mzML || parsed.indexedmzML.mzML;
  processSpectrumList$1(mzML, result.times, result.series.ms.data);
  return result;
}

function processSpectrumList(parsed, times, msData) {
  if (!parsed.msRun.scan) return;
  let scanList = parsed.msRun.scan;
  if (Array.isArray(scanList) === false) scanList = [scanList];
  if (scanList[0].attributes) msData.info = [];
  for (let scan of scanList) {
    if (typeof scan !== 'object') continue;
    if (Array.isArray(scan)) {
      throw new Error('processSpectrumList: scan may not be an array');
    }
    const dataArray = scan.peaks['#text'];
    let length = dataArray.length / 2;
    let first = new Float64Array(length);
    let second = new Float64Array(length);
    for (let i = 0; i < length; i++) {
      first[i] = dataArray[i * 2];
      second[i] = dataArray[i * 2 + 1];
    }
    msData.data.push([first, second]);
    msData.info.push(scan.attributes);
    times.push(parseFloat(scan.attributes.retentionTime.replace(/(?:P*)(?:T*)(?:S*)/gi, '')));
  }
}

const decoder$1 = new TextDecoder();

/**
 *
 * @param {*} arrayBuffer
 * @param {import('../Options.js').Options} [options]
 * @returns
 */
async function parseMzXML(arrayBuffer, options = {}) {
  const {
    logger = console
  } = options;
  const result = {
    metadata: {},
    times: [],
    series: {
      ms: {
        data: []
      }
    }
  };
  let parsed = parse(arrayBuffer, {
    attributesNodeName: 'attributes',
    attributeNameProcessor: attributeName => attributeName,
    tagValueProcessor: (value, node) => {
      if (node.tagName !== 'peaks') return decoder$1.decode(value);
      const promise = decodeData(node.bytes, {
        precision: node.attributes.precision,
        endian: node.attributes.byteOrder,
        compression: node.attributes.compressionType
      });
      // avoid unhandled promise rejection and swallow the error
      promise.catch(error => {
        logger.error('error decoding base64', error);
        return [];
      });
      return promise;
    }
  });
  await recursiveResolve(parsed);
  processSpectrumList(parsed.mzXML, result.times, result.series.ms);
  return result;
}

const decoder = new TextDecoder();

/**
 * Reads a mzData v1.05 file
 * @param {ArrayBuffer|string} xml - ArrayBuffer or String or any Typed Array (including Node.js' Buffer from v4) with the data
 * @param {import('./Options.js').Options} [options={}]
 * @return Promise<{{times: Array<number>, series: { ms: { data:Array<Array<number>>}}}}>
 */
async function parseMZ(xml, options = {}) {
  if (typeof xml === 'string') {
    const encoder = new TextEncoder();
    xml = encoder.encode(xml);
  }
  if (!ArrayBuffer.isView(xml)) {
    xml = new Uint8Array(xml);
  }
  const header = xml.subarray ? decoder.decode(xml.subarray(0, 200)) : xml.substring(0, 200);
  if (header.includes('mzData')) {
    return parseMzData(xml, options);
  } else if (header.includes('mzML')) {
    return parseMzML(xml, options);
  } else if (header.includes('mzXML')) {
    return parseMzXML(xml, options);
  } else {
    throw new Error(`MZ parser: unknown format`);
  }
}

export { parseMZ };
//# sourceMappingURL=mzdata.esm.js.map
