/*
 * screening v2.0.0
 * Tools for numbering screening plate
 * https://github.com/cheminfo-js/screning
 *
 * Licensed under the MIT license.
 */
/**
 * Convert number like 1 to string like 'A'
 * @param number
 * @returns
 */
function numberToString(number) {
  let string = '';
  while (number !== 0) {
    string = String.fromCodePoint((number - 1) % 26 + 65) + string;
    number = Math.floor((number - 1) / 26);
  }
  return string;
}

/**
 * Convert number like 5 to plate position 'A5'
 * @param number
 * @param width
 * @returns
 */
function numberToPosition(number, width) {
  if (width === undefined) {
    throw new Error('need to specify width for numberToPosition');
  }
  number--;
  return numberToString(Math.floor(number / width) + 1) + String(number % width + 1);
}

/**
 * Convert string like 'A' to number like 1
 * @param string
 * @returns
 */
function stringToNumber(string) {
  let number = 0;
  for (let i = 0; i < string.length; i++) {
    number *= 26;
    number += Number(string.codePointAt(i)) - 64;
  }
  return number;
}

/**
 * Convert plate position like 'A5' to 5
 * @param position
 * @param width
 * @returns number
 */
function positionToNumber(position, width) {
  if (width === undefined) {
    throw new Error('need to specify width for numberToPosition');
  }
  position = position.toUpperCase().replaceAll(/[^0-9A-Z]/g, '');
  const string = position.replace(/[0-9]+/, '');
  const number = Number(position.replace(/[A-Z]+/, ''));
  return (stringToNumber(string) - 1) * width + number * 1;
}

class Plate {
  width;
  height;
  size;
  wells = [];
  constructor(options = {}) {
    this.width = options.width ?? 12;
    this.height = options.height ?? this.width / 3 * 2;
    this.size = this.width * this.height;
    this.initialize();
  }
  getWells() {
    return this.wells;
  }
  select(range) {
    if (range) {
      // reset all
      for (const well of this.wells) {
        well.selected = false;
      }
      const reg = /(\d+)(-(\d+))?,?/g;
      let m;
      while (m = reg.exec(range)) {
        const from = Number(m[1]);
        const to = m[3] ? Number(m[3]) : undefined;
        if (to && from <= to) {
          for (let i = from; i <= to; i++) {
            const well = this.wells[i - 1];
            if (!isEmpty(well.info)) well.selected = true;
          }
        } else {
          const well = this.wells[from - 1];
          if (!isEmpty(well.info)) well.selected = true;
        }
      }
    } else {
      // auto-select wells with info
      for (const well of this.wells) {
        well.selected = !isEmpty(well.info);
      }
    }
  }
  getArrayElement(index) {
    return this.wells[index];
  }
  updateColor() {
    for (const well of this.wells) {
      if (isEmpty(well.info)) {
        // empty well = white
        well.info.color = 'white';
      } else if (well.selected) {
        // selected = strong highlight
        well.info.color = 'rgba(144, 238, 144, 1)';
      } else {
        // unselected = faded highlight
        well.info.color = 'rgba(144, 238, 144, 0.3)';
      }
    }
  }
  getByPosition(position) {
    const index = positionToNumber(position, this.width) - 1;
    return this.wells[index];
  }
  getByNumber(number) {
    return this.wells[number - 1];
  }
  initialize() {
    this.wells = new Array(this.size);
    for (let row = 0; row < this.height; row++) {
      for (let column = 0; column < this.width; column++) {
        const i = row * this.width + column;
        this.wells[i] = {
          number: i + 1,
          position: numberToPosition(i + 1, this.width),
          info: {}
        };
      }
    }
  }
}
// ----------- Helpers -----------
function isEmpty(object) {
  return object && typeof object === 'object' && !Array.isArray(object) && Object.keys(object).length === 0;
}

/**
 * Generate iconNMR text file from requests
 * @param requests - array of PlateRequest objects
 * @param options - options for generating the file
 * @returns iconNMR text file
 */
function generateIconNMRFile(requests, options = {}) {
  const {
    eol = '\r\n',
    deleteExistingHolder = false,
    autoSubmit = false
  } = options;
  const textFile = [];
  for (const request of requests) {
    let experimentNumberCounter = 1;
    const {
      holder
    } = request;
    if (deleteExistingHolder) {
      textFile.push(`USER ${request.user}`, `HOLDER ${holder}`, 'DELETE'); // this is required to delete already existing entries
    }
    textFile.push(`USER ${request.user}`, `HOLDER ${holder}`);
    if (!autoSubmit) textFile.push('NO_SUBMIT');
    textFile.push(`NAME ${request.name}`, `TITLE ${request.title}`);
    for (const experiment of request.experiments) {
      let experimentNumber = experiment.experimentNumber || experimentNumberCounter++;
      textFile.push(`EXPNO ${experimentNumber++}`, `SOLVENT ${experiment.solvent}`, `EXPERIMENT ${experiment.experiment}`);
      if (experiment.parameters && experiment.parameters.length > 0) {
        const parameters = [];
        for (const parameter of experiment.parameters) {
          parameters.push(parameter.label, parameter.value);
        }
        textFile.push(`PARAMETERS ${parameters.join(',')}`);
      }
    }
    textFile.push('');
  }
  return textFile.join(eol);
}

/**
 * Generate iconNMR text file from requests with plate positions
 * @param requests - array of PlateRequest objects
 * @param options - options for generating the file
 * @returns iconNMR text file
 */
function generateIconNMRFileFromPlate(requests, options = {}) {
  const {
    plateNumber = 1
  } = options;
  const holderRequests = [];
  for (const request of requests) {
    const {
      position,
      holder,
      experiments,
      ...rest
    } = request;
    const newHolder = holder ?? getHolder(position, plateNumber);
    let experimentNumber = getExperimentNumber(position);
    const experimentsWithHolder = [];
    for (const experiment of experiments) {
      experimentsWithHolder.push({
        experimentNumber: experimentNumber++,
        ...experiment
      });
    }
    holderRequests.push({
      holder: newHolder,
      experiments: experimentsWithHolder,
      ...rest
    });
  }
  return generateIconNMRFile(holderRequests, options);
}
function getHolder(position, plateNumber) {
  return plateNumber * 100 + positionToNumber(position, 12);
}
function getExperimentNumber(position) {
  return positionToNumber(position, 12) * 10;
}

export { Plate, generateIconNMRFile, generateIconNMRFileFromPlate, numberToPosition, numberToString, positionToNumber, stringToNumber };
//# sourceMappingURL=screening.esm.js.map
