/**
 * epfl-safety - Calculate hazard for a reaction
 * @version v1.3.0
 * @link https://github.com/cheminfo/epfl-safety#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.EPFLSafety = {}));
})(this, (function (exports) { 'use strict';

    // compatibility for CustomEvent in Node 16 and 18
    // todo should be removed in 2026 ...
    // https://stackoverflow.com/questions/69791262/how-do-i-send-a-customevent-via-a-node-16-eventtarget
    class CustomEvent extends Event {
      constructor(message, data) {
        super(message, data);
        this.detail = data.detail;
      }
    }

    class LogEvent extends CustomEvent {
      constructor(data) {
        super('log', {
          detail: data
        });
      }
    }
    class ChangeEvent extends CustomEvent {
      constructor(data) {
        super('change', {
          detail: data
        });
      }
    }

    var IDX = 256,
      HEX = [],
      BUFFER;
    while (IDX--) HEX[IDX] = (IDX + 256).toString(16).substring(1);
    function v4() {
      var i = 0,
        num,
        out = '';
      if (!BUFFER || IDX + 16 > 256) {
        BUFFER = Array(i = 256);
        while (i--) BUFFER[i] = 256 * Math.random() | 0;
        i = IDX = 0;
      }
      for (; i < 16; i++) {
        num = BUFFER[IDX + i];
        if (i == 6) out += HEX[num & 15 | 64];else if (i == 8) out += HEX[num & 63 | 128];else out += HEX[num];
        if (i & 1 && i > 1 && i < 11) out += '-';
      }
      IDX++;
      return out;
    }

    var e = class extends EventTarget {
      dispatchTypedEvent(s, t) {
        return super.dispatchEvent(t);
      }
    };

    const levels = {
      values: {
        fatal: 60,
        error: 50,
        warn: 40,
        info: 30,
        debug: 20,
        trace: 10,
        silent: 0
      },
      labels: {
        0: 'silent',
        10: 'trace',
        20: 'debug',
        30: 'info',
        40: 'warn',
        50: 'error',
        60: 'fatal'
      }
    };

    var __classPrivateFieldGet = undefined && undefined.__classPrivateFieldGet || function (receiver, state, kind, f) {
      if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a getter");
      if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
      return kind === "m" ? f : kind === "a" ? f.call(receiver) : f ? f.value : state.get(receiver);
    };
    var _FifoLogger_instances, _FifoLogger_addEvent;
    /**
     * A FIFO logger that stores the last events in an array.
     */
    class FifoLogger extends e {
      constructor(options = {}) {
        super();
        _FifoLogger_instances.add(this);
        this.lastID = {
          id: 0
        };
        this.initialOptions = options;
        this.uuids = [v4()];
        this.events = [];
        this.level = options.level || 'info';
        this.levelAsNumber = levels.values[this.level];
        this.limit = options.limit ?? 1000;
        this.bindings = options.bindings ?? {};
      }
      setLevel(level) {
        this.level = level;
        this.levelAsNumber = levels.values[level];
      }
      setLimit(limit) {
        this.limit = limit;
        this.checkSize();
      }
      checkSize() {
        if (this.events.length > this.limit) {
          this.events.splice(0, this.events.length - this.limit);
        }
      }
      /**
       * Remove events from the current logger and its children.
       */
      clear() {
        for (let i = this.events.length - 1; i >= 0; i--) {
          if (this.events[i].uuids.includes(this.uuids[0])) {
            this.events.splice(i, 1);
          }
        }
        this.dispatchTypedEvent('change', new ChangeEvent({
          logs: this.events,
          info: {
            depth: this.uuids.length
          }
        }));
      }
      /**
       * Get a filtered list of all log events.
       * @param options
       */
      getLogs(options = {}) {
        const {
          level,
          minLevel,
          includeChildren
        } = options;
        let logs = this.events.slice();
        if (includeChildren) {
          logs = logs.filter(log => log.uuids.includes(this.uuids[0]));
        } else {
          logs = logs.filter(log => log.uuids[0] === this.uuids[0]);
        }
        if (level) {
          const levelNumber = Number(levels.values[level]);
          if (Number.isNaN(levelNumber)) {
            throw new Error('Invalid level');
          }
          logs = logs.filter(log => log.level === levelNumber);
        }
        if (minLevel) {
          const levelNumber = Number(levels.values[minLevel]);
          if (Number.isNaN(levelNumber)) {
            throw new Error('Invalid level');
          }
          logs = logs.filter(log => log.level >= levelNumber);
        }
        return logs;
      }
      /**
       * @param bindings an object of key-value pairs to include in log lines as properties.
       */
      child(bindings) {
        const newFifoLogger = new FifoLogger(this.initialOptions);
        newFifoLogger.events = this.events;
        newFifoLogger.uuids = [v4(), ...this.uuids];
        newFifoLogger.lastID = this.lastID;
        newFifoLogger.bindings = {
          ...this.bindings,
          ...bindings
        };
        newFifoLogger.addEventListener('change', event => {
          this.dispatchTypedEvent('change', new ChangeEvent(event.detail));
        });
        return newFifoLogger;
      }
      trace(value, message) {
        __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.trace, value, message);
      }
      debug(value, message) {
        __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.debug, value, message);
      }
      info(value, message) {
        __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.info, value, message);
      }
      warn(value, message) {
        __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.warn, value, message);
      }
      error(value, message) {
        __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.error, value, message);
      }
      fatal(value, message) {
        __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.fatal, value, message);
      }
    }
    _FifoLogger_instances = new WeakSet(), _FifoLogger_addEvent = function _FifoLogger_addEvent(level, value, message) {
      if (level < this.levelAsNumber) return;
      const event = {
        id: ++this.lastID.id,
        level,
        levelLabel: levels.labels[level],
        time: Date.now(),
        uuids: this.uuids,
        message: ''
      };
      if (value instanceof Error) {
        event.message = value.toString();
        event.error = value;
        event.meta = {
          ...this.bindings
        };
      } else if (message !== undefined && typeof value === 'object') {
        event.message = message;
        event.meta = {
          ...this.bindings,
          ...value
        };
      } else if (message === undefined && typeof value === 'string') {
        event.message = value;
        event.meta = {
          ...this.bindings
        };
      } else {
        throw new Error('Invalid arguments');
      }
      this.events.push(event);
      this.checkSize();
      this.dispatchTypedEvent('log', new LogEvent({
        log: event,
        logs: this.events,
        info: {
          depth: this.uuids.length
        }
      }));
      this.dispatchTypedEvent('change', new ChangeEvent({
        logs: this.events,
        info: {
          depth: this.uuids.length
        }
      }));
    };

    const redAcuteToxicitiesHazards = new Set(['H300', 'H310', 'H330']);
    const rules$3 = [{
      description: 'Extreme acute toxicity',
      info: {
        highOrExtreme: true
      },
      matcher: {
        hazards: ['H300', 'H310', 'H330']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 6
      }, {
        lessThan: +Infinity,
        score: 12
      }]
    }, {
      description: 'High acute toxicity',
      info: {
        highOrExtreme: true
      },
      matcher: {
        hazards: ['H311', 'H318', 'H331']
      },
      masses: [{
        lessThan: 0.001,
        score: 2
      }, {
        lessThan: 0.003,
        score: 3
      }, {
        lessThan: +Infinity,
        score: 4
      }]
    }, {
      description: 'Medium acute toxicity',
      info: {
        highOrExtreme: false
      },
      matcher: {
        hazards: ['H302', 'H303', 'H305', 'H312', 'H313', 'H330', 'H332', 'H333', 'H334']
      },
      masses: [{
        lessThan: 0.001,
        score: 1
      }, {
        lessThan: 0.003,
        score: 2
      }, {
        lessThan: +Infinity,
        score: 3
      }]
    }, {
      description: 'Low acute toxicity',
      info: {
        highOrExtreme: false
      },
      matcher: {
        hazards: ['H301', 'H304', 'H315', 'H316', 'H317', 'H319,H320', 'H334', 'H335', 'H336']
      },
      masses: [{
        lessThan: 0.001,
        score: 0.5
      }, {
        lessThan: 0.003,
        score: 1
      }, {
        lessThan: +Infinity,
        score: 2
      }]
    }, {
      description: 'No known acute toxicity',
      info: {
        highOrExtreme: false
      },
      matcher: {},
      masses: [{
        lessThan: +Infinity,
        score: 0
      }]
    }];
    function appendAcuteToxicity(reagents, report, options) {
      const {
        logger
      } = options;
      let redAcuteToxicities = new Set();
      // what is the worse hazard ?
      for (const rule of rules$3) {
        for (const reagent of reagents) {
          if (reagent.safety.acuteToxicity) continue;
          appendRed$1(reagent, redAcuteToxicities);
          const matching = isMatching$3(reagent, rule);
          if (!matching) continue;
          reagent.safety.acuteToxicity = {
            description: rule.description,
            matchingHazards: matching.matchingHazards || [],
            info: rule.info,
            score: rule.masses.find(mass => reagent.mass.value <= mass.lessThan).score
          };
        }
      }

      /*
      If high or extreme are present only take sum of high and extreme values.
      If no high or extreme, simply take the score for the one reagent that leads to the highest score.
      */
      const highOrExtremes = reagents.filter(reagent => reagent.safety.acuteToxicity.info.highOrExtreme);
      if (highOrExtremes.length === 0) {
        logger?.info('AcuteToxicity: No high or extreme acute toxicity. Taking the highest score of the other.');
        report.scores.acuteToxicity = Math.max(...reagents.map(reagent => reagent.safety.acuteToxicity.score));
      } else {
        logger?.info('AcuteToxicity: High or extreme acute toxicity. Taking the sum of them.');
        report.scores.acuteToxicity = highOrExtremes.reduce((sum, reagent) => sum + reagent.safety.acuteToxicity.score, 0);
      }
      report.redAcuteToxicity = [...redAcuteToxicities];
    }
    function isMatching$3(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }
    function appendRed$1(reagent, redAcuteToxicities) {
      for (const hazard of reagent.hazards) {
        if (redAcuteToxicitiesHazards.has(hazard)) {
          redAcuteToxicities.add(hazard);
        }
      }
    }

    const rules$2 = [{
      description: 'Extreme long-term toxicity',
      matcher: {
        hazards: ['H340', 'H350', 'H360', 'H370']
      },
      volumes: [{
        lessThan: 0.05,
        score: 2
      }, {
        lessThan: 0.5,
        score: 3
      }, {
        lessThan: 1,
        score: 6
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'High long-term toxicity',
      matcher: {
        hazards: ['H341', 'H351', 'H361', 'H371', 'H372', 'H373']
      },
      volumes: [{
        lessThan: 0.05,
        score: 1
      }, {
        lessThan: 0.5,
        score: 2
      }, {
        lessThan: 1,
        score: 4
      }, {
        lessThan: +Infinity,
        score: 8
      }]
    }, {
      description: 'No known long-term toxicity',
      matcher: {},
      volumes: [{
        lessThan: +Infinity,
        score: 0
      }]
    }];
    function appendLongTermToxicity(reagents, report, options = {}) {
      const {
        logger
      } = options;
      // what is the worse hazard ?
      for (const rule of rules$2) {
        for (const reagent of reagents) {
          if (reagent.safety.longTermToxicity) continue;
          const matching = isMatching$2(reagent, rule);
          if (!matching) continue;
          reagent.safety.longTermToxicity = {
            description: rule.description,
            matchingHazards: matching.matchingHazards || [],
            score: rule.volumes.find(volume => reagent.volume.value <= volume.lessThan).score
          };
        }
      }
      logger?.info('LongTermToxicity: Taking the highest score.');
      report.scores.longTermToxicity = Math.max(...reagents.map(reagent => reagent.safety.longTermToxicity.score));
    }
    function isMatching$2(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }

    function appendMiscellaneous(reagents, report, options = {}) {
      const {
        miscellaneous = {}
      } = options;
      const {
        highVoltage,
        dustRisk,
        reactionTime,
        temperature,
        gasLines,
        pressure,
        overheadStirring
      } = miscellaneous;
      const results = [];
      results.push(getDustRisk(dustRisk, options));
      results.push(getReactionTime(reactionTime, options));
      results.push(getTemperature(temperature, options));
      results.push(getGasLines(gasLines, options));
      results.push(getPressure(pressure, options));
      results.push(getOverheadStirring(overheadStirring, options));
      results.push(getHighVoltage(highVoltage, options));
      report.miscellaneous = results;
    }
    function getDustRisk(dustRisk, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Dust risk',
        info: '(for more than 5g of solids)',
        scores: []
      };
      switch (dustRisk) {
        case 'yes':
          entry.scores.push({
            remarks: 'Mask still required for all quantities of fine particles!',
            label: 'Fine powders, fibers air suspended',
            score: 4
          });
          break;
        case 'none':
          entry.scores.push({
            label: 'None',
            score: 0
          });
          break;
        default:
          logger.error(`Unknown dustRisk: ${dustRisk}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getReactionTime(reactionTime, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Reaction time',
        info: '',
        scores: []
      };
      switch (reactionTime) {
        case 'daytime':
          entry.scores.push({
            label: 'Daytime',
            score: 0
          });
          break;
        case 'overnight':
          entry.scores.push({
            label: 'Overnight',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown reactionTime: ${reactionTime}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getTemperature(temperature, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Temperature',
        info: '',
        scores: []
      };
      switch (temperature) {
        case '<100':
          entry.scores.push({
            label: '< 100°C',
            score: 0
          });
          break;
        case '100-200':
          entry.scores.push({
            label: '100-200°C',
            score: 1
          });
          break;
        case 'extreme':
          entry.scores.push({
            label: '> 200°C or < -30°C',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown temperature: ${temperature}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getGasLines(gasLines, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Handling High Pressure Gas Lines',
        info: '(Gas line pressure)',
        scores: []
      };
      const inertDescription = 'Inert (e.g., N2, He, Ar, CO2)';
      switch (gasLines.inert) {
        case 'none':
          entry.scores.push({
            description: inertDescription,
            label: 'None',
            score: 0
          });
          break;
        case '<20':
          entry.scores.push({
            description: inertDescription,
            label: '<20 barg',
            score: 0
          });
          break;
        case '20-40':
          entry.scores.push({
            description: inertDescription,
            label: '20-40 barg',
            score: 1
          });
          break;
        case '>40':
          entry.scores.push({
            description: inertDescription,
            label: '>40 barg',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown gasLines.inert: ${gasLines.inert}`);
      }
      const reactiveDescription = 'Reactive/Hazardous (e.g., CO, NH3, H2, hydrocarbons)';
      switch (gasLines.reactive) {
        case 'none':
          entry.scores.push({
            description: reactiveDescription,
            label: 'None',
            score: 0
          });
          break;
        case '<20':
          entry.scores.push({
            description: reactiveDescription,
            label: '<20 barg',
            score: 2
          });
          break;
        case '20-40':
          entry.scores.push({
            description: reactiveDescription,
            label: '20-40 barg',
            score: 3
          });
          break;
        case '>40':
          entry.scores.push({
            description: reactiveDescription,
            label: '>40 barg',
            score: 4
          });
          break;
        default:
          logger.error(`Unknown gasLines.inert: ${gasLines.reactive}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getPressure(pressure, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'For pressurized reactions in Parr/flow',
        info: 'Did you pass the PARR / Flow reactor exam and sign the pressurized SOP and PHA?',
        scores: []
      };
      const parrLowDescription = `Parr < 50 mL or flow reactor (1/4'')`;
      switch (pressure.parrLow) {
        case 'none':
          entry.scores.push({
            description: parrLowDescription,
            label: 'None',
            score: 0
          });
          break;
        case '1-20':
          entry.scores.push({
            description: parrLowDescription,
            label: '1-20 barg',
            score: 1
          });
          break;
        case '20-100':
          entry.scores.push({
            description: parrLowDescription,
            label: '20-100 barg',
            score: 2
          });
          break;
        case '100-120':
          entry.scores.push({
            description: parrLowDescription,
            label: '100-120 barg',
            score: 12
          });
          break;
        case '>120':
          entry.scores.push({
            description: parrLowDescription,
            label: '>120 barg',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.parrLow: ${pressure.parrLow}`);
      }
      const parrHighDescription = `Parr > 50 mL or flow reactor (1/2'')`;
      switch (pressure.parrHigh) {
        case 'none':
          entry.scores.push({
            description: parrHighDescription,
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            description: parrHighDescription,
            label: 'Yes',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.parrHigh: ${pressure.parrHigh}`);
      }
      const pRatedDescription = 'Glass (P-Rated)';
      switch (pressure.pRated) {
        case 'none':
          entry.scores.push({
            description: pRatedDescription,
            label: 'None',
            score: 0
          });
          break;
        case '1-3':
          entry.scores.push({
            description: pRatedDescription,
            label: '1-3 barg',
            score: 1
          });
          break;
        case '3-5':
          entry.scores.push({
            description: pRatedDescription,
            label: '3-5 barg',
            score: 12
          });
          break;
        case '5-8':
          entry.scores.push({
            description: pRatedDescription,
            label: '5-8 barg',
            score: 18
          });
          break;
        case '>8':
          entry.scores.push({
            description: pRatedDescription,
            label: '>8 barg',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.pRated: ${pressure.pRated}`);
      }
      const nonRatedDescription = 'Glass (Not rated)';
      switch (pressure.nonRated) {
        case 'none':
          entry.scores.push({
            description: nonRatedDescription,
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            description: nonRatedDescription,
            label: 'Yes',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.nonRated: ${pressure.nonRated}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getOverheadStirring(overheadStirring, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Overheads stirring ?',
        info: '',
        scores: []
      };
      switch (overheadStirring) {
        case 'none':
          entry.scores.push({
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            label: 'Yes',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown overheadStirring: ${overheadStirring}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getHighVoltage(highVoltage, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'High voltage ?',
        info: '',
        scores: []
      };
      switch (highVoltage) {
        case 'none':
          entry.scores.push({
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            label: 'Yes',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown highVoltage: ${highVoltage}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }

    const redReactivityAndInstabilityHazards = new Set(['H260', 'H261', 'H250', 'H271', 'H272', 'H200', 'H201', 'H202', 'H203', 'H240', 'H241', 'H242', 'H251', 'H252']);
    const rules$1 = [{
      description: 'Flammable gas generation',
      matcher: {
        hazards: ['H260', 'H261']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 5
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Pyrophoric',
      matcher: {
        hazards: ['H250']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 6
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Oxidizer',
      matcher: {
        hazards: ['H270', 'H271', 'H272']
      },
      masses: [{
        lessThan: 0.001,
        score: 1
      }, {
        lessThan: 0.003,
        score: 3
      }, {
        lessThan: +Infinity,
        score: 6
      }]
    }, {
      description: 'Explosive / Self-heating',
      matcher: {
        hazards: ['H200', 'H201', 'H202', 'H203', 'H240', 'H241', 'H242', 'H251', 'H252']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 5
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Corrosive',
      matcher: {
        hazards: ['H314']
      },
      masses: [{
        lessThan: 0.001,
        score: 0.5
      }, {
        lessThan: 0.003,
        score: 1
      }, {
        lessThan: +Infinity,
        score: 2
      }]
    }, {
      description: 'No reactivity or instability hazards known',
      matcher: {},
      masses: [{
        lessThan: +Infinity,
        score: 0
      }]
    }];
    function appendReactivityAndInstability(reagents, report) {
      // what is the worse hazard ?
      let sum = 0;
      let redReactivityAndInstabilities = new Set();
      for (const rule of rules$1) {
        for (const reagent of reagents) {
          if (!reagent.safety.reactivityAndInstabilities) {
            reagent.safety.reactivityAndInstabilities = [];
          }
          const matching = isMatching$1(reagent, rule);
          appendRed(reagent, redReactivityAndInstabilities);
          if (!matching) continue;
          const score = rule.masses.find(mass => reagent.mass.value <= mass.lessThan).score;
          if (score > 0 || reagent.safety.reactivityAndInstabilities.length === 0) {
            reagent.safety.reactivityAndInstabilities.push({
              description: rule.description,
              matchingHazards: matching.matchingHazards || [],
              score
            });
            sum += score;
          }
        }
      }
      report.redReactivityAndInstabilities = [...redReactivityAndInstabilities];
      report.scores.reactivityAndInstability = sum;
    }
    function appendRed(reagent, redReactivity) {
      for (const hazard of reagent.hazards) {
        if (redReactivityAndInstabilityHazards.has(hazard)) {
          redReactivity.add(hazard);
        }
      }
    }
    function isMatching$1(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }

    const rules = [{
      description: 'Very flammable, low bp',
      matcher: {
        hazards: ['H222', 'H224', 'H225'],
        bpLessThan: 35
      },
      volumes: [{
        lessThan: 0.01,
        score: 1.5
      }, {
        lessThan: 0.05,
        score: 3
      }, {
        lessThan: 0.5,
        score: 4
      }, {
        lessThan: 1,
        score: 5
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Very flammable, high bp',
      matcher: {
        hazards: ['H222', 'H224', 'H225']
      },
      volumes: [{
        lessThan: 0.01,
        score: 1
      }, {
        lessThan: 0.05,
        score: 2
      }, {
        lessThan: 0.5,
        score: 3
      }, {
        lessThan: 1,
        score: 4
      }, {
        lessThan: +Infinity,
        score: 8
      }]
    }, {
      description: 'Flammable, low bp',
      matcher: {
        hazards: ['H223', 'H226', 'H227', 'H228'],
        bpLessThan: 35
      },
      volumes: [{
        lessThan: 0.01,
        score: 0.5
      }, {
        lessThan: 0.05,
        score: 1
      }, {
        lessThan: 0.5,
        score: 2
      }, {
        lessThan: 1,
        score: 3
      }, {
        lessThan: +Infinity,
        score: 6
      }]
    }, {
      description: 'Flammable, high bp',
      matcher: {
        hazards: ['H223', 'H226', 'H227', 'H228']
      },
      volumes: [{
        lessThan: 0.01,
        score: 0
      }, {
        lessThan: 0.05,
        score: 0.5
      }, {
        lessThan: 0.5,
        score: 1
      }, {
        lessThan: 1,
        score: 2
      }, {
        lessThan: +Infinity,
        score: 4
      }]
    }, {
      description: 'Not flammable',
      matcher: {},
      volumes: [{
        lessThan: 0.5,
        score: 0
      }, {
        lessThan: 1,
        score: 1
      }, {
        lessThan: +Infinity,
        score: 3
      }]
    }];
    function appendVolumeAndFlammability(reagents, report, options) {
      const totalVolume = getTotalVolume(reagents, options);
      // what is the worse hazard ?
      let bestScore = 0;
      for (const rule of rules) {
        for (const reagent of reagents) {
          if (reagent.safety.volumeAndFlammability) continue;
          const matching = isMatching(reagent, rule);
          if (!matching) continue;
          const score = rule.volumes.find(volume => totalVolume <= volume.lessThan).score;
          if (score > bestScore) {
            bestScore = score;
          }
          reagent.safety.volumeAndFlammability = {
            description: rule.description,
            matchingHazards: matching.matchingHazards || [],
            score
          };
        }
      }
      report.scores.volumeAndFlammability = bestScore;
    }
    function isMatching(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      if (!rule.matcher.bpLessThan && reagent.bp > rule.matcher.bpLessThan) {
        return false;
      }
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }
    function getTotalVolume(reagents, options) {
      const {
        logger
      } = options;
      let totalVolume = 0;
      for (const reagent of reagents) {
        if (reagent.volume) {
          totalVolume += reagent.volume.value;
        } else {
          logger.warn(`reagent ${reagent.smiles} has no volume`);
          totalVolume += reagent.mass.value;
        }
      }
      return totalVolume;
    }

    function filterReagents(reagents, options) {
      const reagentsKind = options.miscellaneous?.reagentsKind;
      if (reagentsKind === 'nonWorkup') {
        reagents = reagents.filter(reagent => reagent.kind?.replace(/[^a-z]/gi, '') !== 'workup');
      } else if (reagentsKind === 'onlyWorkup') {
        reagents = reagents.filter(reagent => reagent.kind?.replace(/[^a-z]/gi, '') === 'workup');
      }
      return reagents;
    }

    // From https://github.com/sindresorhus/random-int/blob/c37741b56f76b9160b0b63dae4e9c64875128146/index.js#L13-L15
    const createAbortError = () => {
      const error = new Error('Delay aborted');
      error.name = 'AbortError';
      return error;
    };
    const clearMethods = new WeakMap();
    function createDelay({
      clearTimeout: defaultClear,
      setTimeout: defaultSet
    } = {}) {
      // We cannot use `async` here as we need the promise identity.
      return (milliseconds, {
        value,
        signal
      } = {}) => {
        // TODO: Use `signal?.throwIfAborted()` when targeting Node.js 18.
        if (signal?.aborted) {
          return Promise.reject(createAbortError());
        }
        let timeoutId;
        let settle;
        let rejectFunction;
        const clear = defaultClear ?? clearTimeout;
        const signalListener = () => {
          clear(timeoutId);
          rejectFunction(createAbortError());
        };
        const cleanup = () => {
          if (signal) {
            signal.removeEventListener('abort', signalListener);
          }
        };
        const delayPromise = new Promise((resolve, reject) => {
          settle = () => {
            cleanup();
            resolve(value);
          };
          rejectFunction = reject;
          timeoutId = (defaultSet ?? setTimeout)(settle, milliseconds);
        });
        if (signal) {
          signal.addEventListener('abort', signalListener, {
            once: true
          });
        }
        clearMethods.set(delayPromise, () => {
          clear(timeoutId);
          timeoutId = null;
          settle();
        });
        return delayPromise;
      };
    }
    const delay = createDelay();

    const ghs = {
      "hStatements": [{
        "code": "H200",
        "statement": "(Deleted) Unstable Explosive"
      }, {
        "code": "H201",
        "statement": "(Deleted) Explosive; mass explosion hazard"
      }, {
        "code": "H202",
        "statement": "(Deleted) Explosive; severe projection hazard"
      }, {
        "code": "H203",
        "statement": "(Deleted) Explosive; fire, blast or projection hazard"
      }, {
        "code": "H204",
        "statement": "Fire or projection hazard"
      }, {
        "code": "H205",
        "statement": "(Deleted) May mass explode in fire"
      }, {
        "code": "H206",
        "statement": "Fire, blast or projection hazard; increased risk of explosion if desensitizing agent is reduced"
      }, {
        "code": "H207",
        "statement": "Fire or projection hazard; increased risk of explosion if desensitizing agent is reduced"
      }, {
        "code": "H208",
        "statement": "Fire hazard; increased risk of explosion if desensitizing agent is reduced"
      }, {
        "code": "H209",
        "statement": "Explosive"
      }, {
        "code": "H210",
        "statement": "Very sensitive"
      }, {
        "code": "H211",
        "statement": "Maybe sensitive"
      }, {
        "code": "H220",
        "statement": "Extremely flammable gas"
      }, {
        "code": "H221",
        "statement": "Flammable gas"
      }, {
        "code": "H222",
        "statement": "Extremely flammable aerosol"
      }, {
        "code": "H223",
        "statement": "Flammable aerosol"
      }, {
        "code": "H224",
        "statement": "Extremely flammable liquid and vapor"
      }, {
        "code": "H225",
        "statement": "Highly Flammable liquid and vapor"
      }, {
        "code": "H226",
        "statement": "Flammable liquid and vapor"
      }, {
        "code": "H227",
        "statement": "Combustible liquid"
      }, {
        "code": "H228",
        "statement": "Flammable solid"
      }, {
        "code": "H229",
        "statement": "Pressurized container: may burst if heated"
      }, {
        "code": "H230",
        "statement": "May react explosively even in the absence of air"
      }, {
        "code": "H231",
        "statement": "May react explosively even in the absence of air at elevated pressure and/or temperature"
      }, {
        "code": "H232",
        "statement": "May ignite spontaneously if exposed to air"
      }, {
        "code": "H240",
        "statement": "Heating may cause an explosion"
      }, {
        "code": "H241",
        "statement": "Heating may cause a fire or explosion"
      }, {
        "code": "H242",
        "statement": "Heating may cause a fire"
      }, {
        "code": "H250",
        "statement": "Catches fire spontaneously if exposed to air"
      }, {
        "code": "H251",
        "statement": "Self-heating; may catch fire"
      }, {
        "code": "H252",
        "statement": "Self-heating in large quantities; may catch fire"
      }, {
        "code": "H260",
        "statement": "In contact with water releases flammable gases which may ignite spontaneously"
      }, {
        "code": "H261",
        "statement": "In contact with water releases flammable gas"
      }, {
        "code": "H270",
        "statement": "May cause or intensify fire; oxidizer"
      }, {
        "code": "H271",
        "statement": "May cause fire or explosion; strong Oxidizer"
      }, {
        "code": "H272",
        "statement": "May intensify fire; oxidizer"
      }, {
        "code": "H280",
        "statement": "Contains gas under pressure; may explode if heated"
      }, {
        "code": "H281",
        "statement": "Contains refrigerated gas; may cause cryogenic burns or injury"
      }, {
        "code": "H282",
        "statement": "Extremely flammable chemical under pressure: may explode if heated"
      }, {
        "code": "H283",
        "statement": "Flammable chemical under pressure: may explode if heated"
      }, {
        "code": "H284",
        "statement": "Chemical under pressure: may explode if heated"
      }, {
        "code": "H290",
        "statement": "May be corrosive to metals"
      }, {
        "code": "H300",
        "statement": "Fatal if swallowed"
      }, {
        "code": "H301",
        "statement": "Toxic if swallowed"
      }, {
        "code": "H302",
        "statement": "Harmful if swallowed"
      }, {
        "code": "H303",
        "statement": "May be harmful if swallowed"
      }, {
        "code": "H304",
        "statement": "May be fatal if swallowed and enters airways"
      }, {
        "code": "H305",
        "statement": "May be harmful if swallowed and enters airways"
      }, {
        "code": "H310",
        "statement": "Fatal in contact with skin"
      }, {
        "code": "H311",
        "statement": "Toxic in contact with skin"
      }, {
        "code": "H312",
        "statement": "Harmful in contact with skin"
      }, {
        "code": "H313",
        "statement": "May be harmful in contact with skin"
      }, {
        "code": "H314",
        "statement": "Causes severe skin burns and eye damage"
      }, {
        "code": "H315",
        "statement": "Causes skin irritation"
      }, {
        "code": "H316",
        "statement": "Causes mild skin irritation"
      }, {
        "code": "H317",
        "statement": "May cause an allergic skin reaction"
      }, {
        "code": "H318",
        "statement": "Causes serious eye damage"
      }, {
        "code": "H319",
        "statement": "Causes serious eye irritation"
      }, {
        "code": "H320",
        "statement": "Causes eye irritation"
      }, {
        "code": "H330",
        "statement": "Fatal if inhaled"
      }, {
        "code": "H331",
        "statement": "Toxic if inhaled"
      }, {
        "code": "H332",
        "statement": "Harmful if inhaled"
      }, {
        "code": "H333",
        "statement": "May be harmful if inhaled"
      }, {
        "code": "H334",
        "statement": "May cause allergy or asthma symptoms or breathing difficulties if inhaled"
      }, {
        "code": "H335",
        "statement": "May cause respiratory irritation"
      }, {
        "code": "H336",
        "statement": "May cause drowsiness or dizziness"
      }, {
        "code": "H340",
        "statement": "May cause genetic defects"
      }, {
        "code": "H341",
        "statement": "Suspected of causing genetic defects"
      }, {
        "code": "H350",
        "statement": "May cause cancer"
      }, {
        "code": "H350i",
        "statement": "May cause cancer by inhalation"
      }, {
        "code": "H351",
        "statement": "Suspected of causing cancer"
      }, {
        "code": "H360",
        "statement": "May damage fertility or the unborn child"
      }, {
        "code": "H360F",
        "statement": "May damage fertility"
      }, {
        "code": "H360D",
        "statement": "May damage the unborn child"
      }, {
        "code": "H360FD",
        "statement": "May damage fertility; May damage the unborn child"
      }, {
        "code": "H360Fd",
        "statement": "May damage fertility; Suspected of damaging the unborn child"
      }, {
        "code": "H360Df",
        "statement": "May damage the unborn child; Suspected of damaging fertility"
      }, {
        "code": "H361",
        "statement": "Suspected of damaging fertility or the unborn child"
      }, {
        "code": "H361f",
        "statement": "Suspected of damaging fertility"
      }, {
        "code": "H361d",
        "statement": "Suspected of damaging the unborn child"
      }, {
        "code": "H361fd",
        "statement": "Suspected of damaging fertility; Suspected of damaging the unborn child"
      }, {
        "code": "H362",
        "statement": "May cause harm to breast-fed children"
      }, {
        "code": "H370",
        "statement": "Causes damage to organs"
      }, {
        "code": "H371",
        "statement": "May cause damage to organs"
      }, {
        "code": "H372",
        "statement": "Causes damage to organs through prolonged or repeated exposure"
      }, {
        "code": "H373",
        "statement": "May causes damage to organs through prolonged or repeated exposure"
      }, {
        "code": "H400",
        "statement": "Very toxic to aquatic life"
      }, {
        "code": "H401",
        "statement": "Toxic to aquatic life"
      }, {
        "code": "H402",
        "statement": "Harmful to aquatic life"
      }, {
        "code": "H410",
        "statement": "Very toxic to aquatic life with long lasting effects"
      }, {
        "code": "H411",
        "statement": "Toxic to aquatic life with long lasting effects"
      }, {
        "code": "H412",
        "statement": "Harmful to aquatic life with long lasting effects"
      }, {
        "code": "H413",
        "statement": "May cause long lasting harmful effects to aquatic life"
      }, {
        "code": "H420",
        "statement": "Harms public health and the environment by destroying ozone in the upper atmosphere"
      }, {
        "code": "H300+H310",
        "statement": "Fatal if swallowed or in contact with skin"
      }, {
        "code": "H300+H330",
        "statement": "Fatal if swallowed or if inhaled"
      }, {
        "code": "H310+H330",
        "statement": "Fatal in contact with skin or if inhaled"
      }, {
        "code": "H300+H310+H330",
        "statement": "Fatal if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H301+H311",
        "statement": "Toxic if swallowed or in contact with skin"
      }, {
        "code": "H301+H331",
        "statement": "Toxic if swallowed or if inhaled"
      }, {
        "code": "H311+H331",
        "statement": "Toxic in contact with skin or if inhaled."
      }, {
        "code": "H301+H311+H331",
        "statement": "Toxic if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H302+H312",
        "statement": "Harmful if swallowed or in contact with skin"
      }, {
        "code": "H302+H332",
        "statement": "Harmful if swallowed or if inhaled"
      }, {
        "code": "H312+H332",
        "statement": "Harmful in contact with skin or if inhaled"
      }, {
        "code": "H302+H312+H332",
        "statement": "Harmful if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H303+H313",
        "statement": "May be harmful if swallowed or in contact with skin"
      }, {
        "code": "H303+H333",
        "statement": "May be harmful if swallowed or if inhaled"
      }, {
        "code": "H313+H333",
        "statement": "May be harmful in contact with skin or if inhaled"
      }, {
        "code": "H303+H313+H333",
        "statement": "May be harmful if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H315+H320",
        "statement": "Cause skin and eye irritation"
      }, {
        "code": "EUH001",
        "statement": "Explosive when dry"
      }, {
        "code": "EUH006",
        "statement": "Explosive with or without contact with air"
      }, {
        "code": "EUH014",
        "statement": "Reacts violently with water"
      }, {
        "code": "EUH018",
        "statement": "In use may form flammable/explosive vapour-air mixture"
      }, {
        "code": "EUH019",
        "statement": "May form explosive peroxides"
      }, {
        "code": "EUH044",
        "statement": "Risk of explosion if heated under confinement"
      }, {
        "code": "EUH029",
        "statement": "Contact with water liberates toxic gas"
      }, {
        "code": "EUH031",
        "statement": "Contact with acids liberates toxic gas"
      }, {
        "code": "EUH032",
        "statement": "Contact with acids liberates very toxic gas"
      }, {
        "code": "EUH066",
        "statement": "Repeated exposure may cause skin dryness or cracking"
      }, {
        "code": "EUH070",
        "statement": "Toxic by eye contact"
      }, {
        "code": "EUH071",
        "statement": "Corrosive to the respiratory tract"
      }, {
        "code": "EUH059",
        "statement": "Hazardous to the ozone layer, superseded by GHS Class 5.1 in the second adaptation to technical progress of CLP."
      }, {
        "code": "EUH201",
        "statement": "Contains lead. Should not be used on surfaces liable to be chewed or sucked by children."
      }, {
        "code": "EUH201A",
        "statement": "Warning! Contains lead."
      }, {
        "code": "EUH202",
        "statement": "Cyanoacrylate. Danger. Bonds skin and eyes in seconds. Keep out of the reach of children."
      }, {
        "code": "EUH203",
        "statement": "Contains chromium(VI). May produce an allergic reaction."
      }, {
        "code": "EUH204",
        "statement": "Contains isocyanates. May produce an allergic reaction."
      }, {
        "code": "EUH205",
        "statement": "Contains epoxy constituents. May produce an allergic reaction."
      }, {
        "code": "EUH206",
        "statement": "Warning! Do not use together with other products. May release dangerous gases (chlorine)."
      }, {
        "code": "EUH207",
        "statement": "Warning! Contains cadmium. Dangerous fumes are formed during use. See information supplied by the manufacturer. Comply with the safety instructions."
      }, {
        "code": "EUH208",
        "statement": "Contains <name of sensitising substance>. May produce an allergic reaction."
      }, {
        "code": "EUH209",
        "statement": "Can become highly flammable in use."
      }, {
        "code": "EUH209A",
        "statement": "Can become flammable in use."
      }, {
        "code": "EUH210",
        "statement": "Safety data sheet available on request."
      }, {
        "code": "EUH401",
        "statement": "To avoid risks to human health and the environment, comply with the instructions for use."
      }]};

    function getHStatementsObject() {
      const result = {};
      ghs.hStatements.forEach(h => {
        if (result[h.code]) {
          throw new Error(`This should never happen, duplice hCode: ${h.code}`);
        }
        result[h.code] = h;
      });
      return result;
    }

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    var browserPonyfill = {exports: {}};

    (function (module, exports) {
      // Save global object in a variable
      var __global__ = typeof globalThis !== 'undefined' && globalThis || typeof self !== 'undefined' && self || typeof commonjsGlobal !== 'undefined' && commonjsGlobal;
      // Create an object that extends from __global__ without the fetch function
      var __globalThis__ = function () {
        function F() {
          this.fetch = false;
          this.DOMException = __global__.DOMException;
        }
        F.prototype = __global__; // Needed for feature detection on whatwg-fetch's code
        return new F();
      }();
      // Wraps whatwg-fetch with a function scope to hijack the global object
      // "globalThis" that's going to be patched
      (function (globalThis) {
        (function (exports) {
          /* eslint-disable no-prototype-builtins */
          var g = typeof globalThis !== 'undefined' && globalThis || typeof self !== 'undefined' && self ||
          // eslint-disable-next-line no-undef
          typeof commonjsGlobal !== 'undefined' && commonjsGlobal || {};
          var support = {
            searchParams: 'URLSearchParams' in g,
            iterable: 'Symbol' in g && 'iterator' in Symbol,
            blob: 'FileReader' in g && 'Blob' in g && function () {
              try {
                new Blob();
                return true;
              } catch (e) {
                return false;
              }
            }(),
            formData: 'FormData' in g,
            arrayBuffer: 'ArrayBuffer' in g
          };
          function isDataView(obj) {
            return obj && DataView.prototype.isPrototypeOf(obj);
          }
          if (support.arrayBuffer) {
            var viewClasses = ['[object Int8Array]', '[object Uint8Array]', '[object Uint8ClampedArray]', '[object Int16Array]', '[object Uint16Array]', '[object Int32Array]', '[object Uint32Array]', '[object Float32Array]', '[object Float64Array]'];
            var isArrayBufferView = ArrayBuffer.isView || function (obj) {
              return obj && viewClasses.indexOf(Object.prototype.toString.call(obj)) > -1;
            };
          }
          function normalizeName(name) {
            if (typeof name !== 'string') {
              name = String(name);
            }
            if (/[^a-z0-9\-#$%&'*+.^_`|~!]/i.test(name) || name === '') {
              throw new TypeError('Invalid character in header field name: "' + name + '"');
            }
            return name.toLowerCase();
          }
          function normalizeValue(value) {
            if (typeof value !== 'string') {
              value = String(value);
            }
            return value;
          }

          // Build a destructive iterator for the value list
          function iteratorFor(items) {
            var iterator = {
              next: function () {
                var value = items.shift();
                return {
                  done: value === undefined,
                  value: value
                };
              }
            };
            if (support.iterable) {
              iterator[Symbol.iterator] = function () {
                return iterator;
              };
            }
            return iterator;
          }
          function Headers(headers) {
            this.map = {};
            if (headers instanceof Headers) {
              headers.forEach(function (value, name) {
                this.append(name, value);
              }, this);
            } else if (Array.isArray(headers)) {
              headers.forEach(function (header) {
                if (header.length != 2) {
                  throw new TypeError('Headers constructor: expected name/value pair to be length 2, found' + header.length);
                }
                this.append(header[0], header[1]);
              }, this);
            } else if (headers) {
              Object.getOwnPropertyNames(headers).forEach(function (name) {
                this.append(name, headers[name]);
              }, this);
            }
          }
          Headers.prototype.append = function (name, value) {
            name = normalizeName(name);
            value = normalizeValue(value);
            var oldValue = this.map[name];
            this.map[name] = oldValue ? oldValue + ', ' + value : value;
          };
          Headers.prototype['delete'] = function (name) {
            delete this.map[normalizeName(name)];
          };
          Headers.prototype.get = function (name) {
            name = normalizeName(name);
            return this.has(name) ? this.map[name] : null;
          };
          Headers.prototype.has = function (name) {
            return this.map.hasOwnProperty(normalizeName(name));
          };
          Headers.prototype.set = function (name, value) {
            this.map[normalizeName(name)] = normalizeValue(value);
          };
          Headers.prototype.forEach = function (callback, thisArg) {
            for (var name in this.map) {
              if (this.map.hasOwnProperty(name)) {
                callback.call(thisArg, this.map[name], name, this);
              }
            }
          };
          Headers.prototype.keys = function () {
            var items = [];
            this.forEach(function (value, name) {
              items.push(name);
            });
            return iteratorFor(items);
          };
          Headers.prototype.values = function () {
            var items = [];
            this.forEach(function (value) {
              items.push(value);
            });
            return iteratorFor(items);
          };
          Headers.prototype.entries = function () {
            var items = [];
            this.forEach(function (value, name) {
              items.push([name, value]);
            });
            return iteratorFor(items);
          };
          if (support.iterable) {
            Headers.prototype[Symbol.iterator] = Headers.prototype.entries;
          }
          function consumed(body) {
            if (body._noBody) return;
            if (body.bodyUsed) {
              return Promise.reject(new TypeError('Already read'));
            }
            body.bodyUsed = true;
          }
          function fileReaderReady(reader) {
            return new Promise(function (resolve, reject) {
              reader.onload = function () {
                resolve(reader.result);
              };
              reader.onerror = function () {
                reject(reader.error);
              };
            });
          }
          function readBlobAsArrayBuffer(blob) {
            var reader = new FileReader();
            var promise = fileReaderReady(reader);
            reader.readAsArrayBuffer(blob);
            return promise;
          }
          function readBlobAsText(blob) {
            var reader = new FileReader();
            var promise = fileReaderReady(reader);
            var match = /charset=([A-Za-z0-9_-]+)/.exec(blob.type);
            var encoding = match ? match[1] : 'utf-8';
            reader.readAsText(blob, encoding);
            return promise;
          }
          function readArrayBufferAsText(buf) {
            var view = new Uint8Array(buf);
            var chars = new Array(view.length);
            for (var i = 0; i < view.length; i++) {
              chars[i] = String.fromCharCode(view[i]);
            }
            return chars.join('');
          }
          function bufferClone(buf) {
            if (buf.slice) {
              return buf.slice(0);
            } else {
              var view = new Uint8Array(buf.byteLength);
              view.set(new Uint8Array(buf));
              return view.buffer;
            }
          }
          function Body() {
            this.bodyUsed = false;
            this._initBody = function (body) {
              /*
                fetch-mock wraps the Response object in an ES6 Proxy to
                provide useful test harness features such as flush. However, on
                ES5 browsers without fetch or Proxy support pollyfills must be used;
                the proxy-pollyfill is unable to proxy an attribute unless it exists
                on the object before the Proxy is created. This change ensures
                Response.bodyUsed exists on the instance, while maintaining the
                semantic of setting Request.bodyUsed in the constructor before
                _initBody is called.
              */
              // eslint-disable-next-line no-self-assign
              this.bodyUsed = this.bodyUsed;
              this._bodyInit = body;
              if (!body) {
                this._noBody = true;
                this._bodyText = '';
              } else if (typeof body === 'string') {
                this._bodyText = body;
              } else if (support.blob && Blob.prototype.isPrototypeOf(body)) {
                this._bodyBlob = body;
              } else if (support.formData && FormData.prototype.isPrototypeOf(body)) {
                this._bodyFormData = body;
              } else if (support.searchParams && URLSearchParams.prototype.isPrototypeOf(body)) {
                this._bodyText = body.toString();
              } else if (support.arrayBuffer && support.blob && isDataView(body)) {
                this._bodyArrayBuffer = bufferClone(body.buffer);
                // IE 10-11 can't handle a DataView body.
                this._bodyInit = new Blob([this._bodyArrayBuffer]);
              } else if (support.arrayBuffer && (ArrayBuffer.prototype.isPrototypeOf(body) || isArrayBufferView(body))) {
                this._bodyArrayBuffer = bufferClone(body);
              } else {
                this._bodyText = body = Object.prototype.toString.call(body);
              }
              if (!this.headers.get('content-type')) {
                if (typeof body === 'string') {
                  this.headers.set('content-type', 'text/plain;charset=UTF-8');
                } else if (this._bodyBlob && this._bodyBlob.type) {
                  this.headers.set('content-type', this._bodyBlob.type);
                } else if (support.searchParams && URLSearchParams.prototype.isPrototypeOf(body)) {
                  this.headers.set('content-type', 'application/x-www-form-urlencoded;charset=UTF-8');
                }
              }
            };
            if (support.blob) {
              this.blob = function () {
                var rejected = consumed(this);
                if (rejected) {
                  return rejected;
                }
                if (this._bodyBlob) {
                  return Promise.resolve(this._bodyBlob);
                } else if (this._bodyArrayBuffer) {
                  return Promise.resolve(new Blob([this._bodyArrayBuffer]));
                } else if (this._bodyFormData) {
                  throw new Error('could not read FormData body as blob');
                } else {
                  return Promise.resolve(new Blob([this._bodyText]));
                }
              };
            }
            this.arrayBuffer = function () {
              if (this._bodyArrayBuffer) {
                var isConsumed = consumed(this);
                if (isConsumed) {
                  return isConsumed;
                } else if (ArrayBuffer.isView(this._bodyArrayBuffer)) {
                  return Promise.resolve(this._bodyArrayBuffer.buffer.slice(this._bodyArrayBuffer.byteOffset, this._bodyArrayBuffer.byteOffset + this._bodyArrayBuffer.byteLength));
                } else {
                  return Promise.resolve(this._bodyArrayBuffer);
                }
              } else if (support.blob) {
                return this.blob().then(readBlobAsArrayBuffer);
              } else {
                throw new Error('could not read as ArrayBuffer');
              }
            };
            this.text = function () {
              var rejected = consumed(this);
              if (rejected) {
                return rejected;
              }
              if (this._bodyBlob) {
                return readBlobAsText(this._bodyBlob);
              } else if (this._bodyArrayBuffer) {
                return Promise.resolve(readArrayBufferAsText(this._bodyArrayBuffer));
              } else if (this._bodyFormData) {
                throw new Error('could not read FormData body as text');
              } else {
                return Promise.resolve(this._bodyText);
              }
            };
            if (support.formData) {
              this.formData = function () {
                return this.text().then(decode);
              };
            }
            this.json = function () {
              return this.text().then(JSON.parse);
            };
            return this;
          }

          // HTTP methods whose capitalization should be normalized
          var methods = ['CONNECT', 'DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT', 'TRACE'];
          function normalizeMethod(method) {
            var upcased = method.toUpperCase();
            return methods.indexOf(upcased) > -1 ? upcased : method;
          }
          function Request(input, options) {
            if (!(this instanceof Request)) {
              throw new TypeError('Please use the "new" operator, this DOM object constructor cannot be called as a function.');
            }
            options = options || {};
            var body = options.body;
            if (input instanceof Request) {
              if (input.bodyUsed) {
                throw new TypeError('Already read');
              }
              this.url = input.url;
              this.credentials = input.credentials;
              if (!options.headers) {
                this.headers = new Headers(input.headers);
              }
              this.method = input.method;
              this.mode = input.mode;
              this.signal = input.signal;
              if (!body && input._bodyInit != null) {
                body = input._bodyInit;
                input.bodyUsed = true;
              }
            } else {
              this.url = String(input);
            }
            this.credentials = options.credentials || this.credentials || 'same-origin';
            if (options.headers || !this.headers) {
              this.headers = new Headers(options.headers);
            }
            this.method = normalizeMethod(options.method || this.method || 'GET');
            this.mode = options.mode || this.mode || null;
            this.signal = options.signal || this.signal || function () {
              if ('AbortController' in g) {
                var ctrl = new AbortController();
                return ctrl.signal;
              }
            }();
            this.referrer = null;
            if ((this.method === 'GET' || this.method === 'HEAD') && body) {
              throw new TypeError('Body not allowed for GET or HEAD requests');
            }
            this._initBody(body);
            if (this.method === 'GET' || this.method === 'HEAD') {
              if (options.cache === 'no-store' || options.cache === 'no-cache') {
                // Search for a '_' parameter in the query string
                var reParamSearch = /([?&])_=[^&]*/;
                if (reParamSearch.test(this.url)) {
                  // If it already exists then set the value with the current time
                  this.url = this.url.replace(reParamSearch, '$1_=' + new Date().getTime());
                } else {
                  // Otherwise add a new '_' parameter to the end with the current time
                  var reQueryString = /\?/;
                  this.url += (reQueryString.test(this.url) ? '&' : '?') + '_=' + new Date().getTime();
                }
              }
            }
          }
          Request.prototype.clone = function () {
            return new Request(this, {
              body: this._bodyInit
            });
          };
          function decode(body) {
            var form = new FormData();
            body.trim().split('&').forEach(function (bytes) {
              if (bytes) {
                var split = bytes.split('=');
                var name = split.shift().replace(/\+/g, ' ');
                var value = split.join('=').replace(/\+/g, ' ');
                form.append(decodeURIComponent(name), decodeURIComponent(value));
              }
            });
            return form;
          }
          function parseHeaders(rawHeaders) {
            var headers = new Headers();
            // Replace instances of \r\n and \n followed by at least one space or horizontal tab with a space
            // https://tools.ietf.org/html/rfc7230#section-3.2
            var preProcessedHeaders = rawHeaders.replace(/\r?\n[\t ]+/g, ' ');
            // Avoiding split via regex to work around a common IE11 bug with the core-js 3.6.0 regex polyfill
            // https://github.com/github/fetch/issues/748
            // https://github.com/zloirock/core-js/issues/751
            preProcessedHeaders.split('\r').map(function (header) {
              return header.indexOf('\n') === 0 ? header.substr(1, header.length) : header;
            }).forEach(function (line) {
              var parts = line.split(':');
              var key = parts.shift().trim();
              if (key) {
                var value = parts.join(':').trim();
                try {
                  headers.append(key, value);
                } catch (error) {
                  console.warn('Response ' + error.message);
                }
              }
            });
            return headers;
          }
          Body.call(Request.prototype);
          function Response(bodyInit, options) {
            if (!(this instanceof Response)) {
              throw new TypeError('Please use the "new" operator, this DOM object constructor cannot be called as a function.');
            }
            if (!options) {
              options = {};
            }
            this.type = 'default';
            this.status = options.status === undefined ? 200 : options.status;
            if (this.status < 200 || this.status > 599) {
              throw new RangeError("Failed to construct 'Response': The status provided (0) is outside the range [200, 599].");
            }
            this.ok = this.status >= 200 && this.status < 300;
            this.statusText = options.statusText === undefined ? '' : '' + options.statusText;
            this.headers = new Headers(options.headers);
            this.url = options.url || '';
            this._initBody(bodyInit);
          }
          Body.call(Response.prototype);
          Response.prototype.clone = function () {
            return new Response(this._bodyInit, {
              status: this.status,
              statusText: this.statusText,
              headers: new Headers(this.headers),
              url: this.url
            });
          };
          Response.error = function () {
            var response = new Response(null, {
              status: 200,
              statusText: ''
            });
            response.ok = false;
            response.status = 0;
            response.type = 'error';
            return response;
          };
          var redirectStatuses = [301, 302, 303, 307, 308];
          Response.redirect = function (url, status) {
            if (redirectStatuses.indexOf(status) === -1) {
              throw new RangeError('Invalid status code');
            }
            return new Response(null, {
              status: status,
              headers: {
                location: url
              }
            });
          };
          exports.DOMException = g.DOMException;
          try {
            new exports.DOMException();
          } catch (err) {
            exports.DOMException = function (message, name) {
              this.message = message;
              this.name = name;
              var error = Error(message);
              this.stack = error.stack;
            };
            exports.DOMException.prototype = Object.create(Error.prototype);
            exports.DOMException.prototype.constructor = exports.DOMException;
          }
          function fetch(input, init) {
            return new Promise(function (resolve, reject) {
              var request = new Request(input, init);
              if (request.signal && request.signal.aborted) {
                return reject(new exports.DOMException('Aborted', 'AbortError'));
              }
              var xhr = new XMLHttpRequest();
              function abortXhr() {
                xhr.abort();
              }
              xhr.onload = function () {
                var options = {
                  statusText: xhr.statusText,
                  headers: parseHeaders(xhr.getAllResponseHeaders() || '')
                };
                // This check if specifically for when a user fetches a file locally from the file system
                // Only if the status is out of a normal range
                if (request.url.indexOf('file://') === 0 && (xhr.status < 200 || xhr.status > 599)) {
                  options.status = 200;
                } else {
                  options.status = xhr.status;
                }
                options.url = 'responseURL' in xhr ? xhr.responseURL : options.headers.get('X-Request-URL');
                var body = 'response' in xhr ? xhr.response : xhr.responseText;
                setTimeout(function () {
                  resolve(new Response(body, options));
                }, 0);
              };
              xhr.onerror = function () {
                setTimeout(function () {
                  reject(new TypeError('Network request failed'));
                }, 0);
              };
              xhr.ontimeout = function () {
                setTimeout(function () {
                  reject(new TypeError('Network request timed out'));
                }, 0);
              };
              xhr.onabort = function () {
                setTimeout(function () {
                  reject(new exports.DOMException('Aborted', 'AbortError'));
                }, 0);
              };
              function fixUrl(url) {
                try {
                  return url === '' && g.location.href ? g.location.href : url;
                } catch (e) {
                  return url;
                }
              }
              xhr.open(request.method, fixUrl(request.url), true);
              if (request.credentials === 'include') {
                xhr.withCredentials = true;
              } else if (request.credentials === 'omit') {
                xhr.withCredentials = false;
              }
              if ('responseType' in xhr) {
                if (support.blob) {
                  xhr.responseType = 'blob';
                } else if (support.arrayBuffer) {
                  xhr.responseType = 'arraybuffer';
                }
              }
              if (init && typeof init.headers === 'object' && !(init.headers instanceof Headers || g.Headers && init.headers instanceof g.Headers)) {
                var names = [];
                Object.getOwnPropertyNames(init.headers).forEach(function (name) {
                  names.push(normalizeName(name));
                  xhr.setRequestHeader(name, normalizeValue(init.headers[name]));
                });
                request.headers.forEach(function (value, name) {
                  if (names.indexOf(name) === -1) {
                    xhr.setRequestHeader(name, value);
                  }
                });
              } else {
                request.headers.forEach(function (value, name) {
                  xhr.setRequestHeader(name, value);
                });
              }
              if (request.signal) {
                request.signal.addEventListener('abort', abortXhr);
                xhr.onreadystatechange = function () {
                  // DONE (success or failure)
                  if (xhr.readyState === 4) {
                    request.signal.removeEventListener('abort', abortXhr);
                  }
                };
              }
              xhr.send(typeof request._bodyInit === 'undefined' ? null : request._bodyInit);
            });
          }
          fetch.polyfill = true;
          if (!g.fetch) {
            g.fetch = fetch;
            g.Headers = Headers;
            g.Request = Request;
            g.Response = Response;
          }
          exports.Headers = Headers;
          exports.Request = Request;
          exports.Response = Response;
          exports.fetch = fetch;
          return exports;
        })({});
      })(__globalThis__);
      // This is a ponyfill, so...
      __globalThis__.fetch.ponyfill = true;
      delete __globalThis__.fetch.polyfill;
      // Choose between native implementation (__global__) or custom implementation (__globalThis__)
      var ctx = __global__.fetch ? __global__ : __globalThis__;
      exports = ctx.fetch; // To enable: import fetch from 'cross-fetch'
      exports.default = ctx.fetch; // For TypeScript consumers without esModuleInterop.
      exports.fetch = ctx.fetch; // To enable: import {fetch} from 'cross-fetch'
      exports.Headers = ctx.Headers;
      exports.Request = ctx.Request;
      exports.Response = ctx.Response;
      module.exports = exports;
    })(browserPonyfill, browserPonyfill.exports);
    var browserPonyfillExports = browserPonyfill.exports;
    var fetch = /*@__PURE__*/getDefaultExportFromCjs(browserPonyfillExports);

    function commonjsRequire(path) {
    	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
    }

    var jsonpath = {exports: {}};

    /*! jsonpath 1.1.1 */
    (function (module, exports) {
      (function (f) {
        {
          module.exports = f();
        }
      })(function () {
        return function e(t, n, r) {
          function s(o, u) {
            if (!n[o]) {
              if (!t[o]) {
                var a = typeof commonjsRequire == "function" && commonjsRequire;
                if (!u && a) return a(o, true);
                if (i) return i(o, true);
                var f = new Error("Cannot find module '" + o + "'");
                throw f.code = "MODULE_NOT_FOUND", f;
              }
              var l = n[o] = {
                exports: {}
              };
              t[o][0].call(l.exports, function (e) {
                var n = t[o][1][e];
                return s(n ? n : e);
              }, l, l.exports, e, t, n, r);
            }
            return n[o].exports;
          }
          var i = typeof commonjsRequire == "function" && commonjsRequire;
          for (var o = 0; o < r.length; o++) s(r[o]);
          return s;
        }({
          "./aesprim": [function (require, module, exports) {
            /*
              Copyright (C) 2013 Ariya Hidayat <ariya.hidayat@gmail.com>
              Copyright (C) 2013 Thaddee Tyl <thaddee.tyl@gmail.com>
              Copyright (C) 2013 Mathias Bynens <mathias@qiwi.be>
              Copyright (C) 2012 Ariya Hidayat <ariya.hidayat@gmail.com>
              Copyright (C) 2012 Mathias Bynens <mathias@qiwi.be>
              Copyright (C) 2012 Joost-Wim Boekesteijn <joost-wim@boekesteijn.nl>
              Copyright (C) 2012 Kris Kowal <kris.kowal@cixar.com>
              Copyright (C) 2012 Yusuke Suzuki <utatane.tea@gmail.com>
              Copyright (C) 2012 Arpad Borsos <arpad.borsos@googlemail.com>
              Copyright (C) 2011 Ariya Hidayat <ariya.hidayat@gmail.com>
            	  Redistribution and use in source and binary forms, with or without
              modification, are permitted provided that the following conditions are met:
            	    * Redistributions of source code must retain the above copyright
                  notice, this list of conditions and the following disclaimer.
                * Redistributions in binary form must reproduce the above copyright
                  notice, this list of conditions and the following disclaimer in the
                  documentation and/or other materials provided with the distribution.
            	  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
              AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
              IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
              ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
              DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
              (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
              LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
              ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
              (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
              THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
            */

            /*jslint bitwise:true plusplus:true */
            /*global esprima:true, define:true, exports:true, window: true,
            throwErrorTolerant: true,
            throwError: true, generateStatement: true, peek: true,
            parseAssignmentExpression: true, parseBlock: true, parseExpression: true,
            parseFunctionDeclaration: true, parseFunctionExpression: true,
            parseFunctionSourceElements: true, parseVariableIdentifier: true,
            parseLeftHandSideExpression: true,
            parseUnaryExpression: true,
            parseStatement: true, parseSourceElement: true */

            (function (root, factory) {

              // Universal Module Definition (UMD) to support AMD, CommonJS/Node.js,
              // Rhino, and plain browser loading.

              /* istanbul ignore next */
              if (typeof exports !== 'undefined') {
                factory(exports);
              } else {
                factory(root.esprima = {});
              }
            })(this, function (exports) {

              var Token, TokenName, FnExprTokens, Syntax, PropertyKind, Messages, Regex, SyntaxTreeDelegate, source, strict, index, lineNumber, lineStart, length, delegate, lookahead, state, extra;
              Token = {
                BooleanLiteral: 1,
                EOF: 2,
                Identifier: 3,
                Keyword: 4,
                NullLiteral: 5,
                NumericLiteral: 6,
                Punctuator: 7,
                StringLiteral: 8,
                RegularExpression: 9
              };
              TokenName = {};
              TokenName[Token.BooleanLiteral] = 'Boolean';
              TokenName[Token.EOF] = '<end>';
              TokenName[Token.Identifier] = 'Identifier';
              TokenName[Token.Keyword] = 'Keyword';
              TokenName[Token.NullLiteral] = 'Null';
              TokenName[Token.NumericLiteral] = 'Numeric';
              TokenName[Token.Punctuator] = 'Punctuator';
              TokenName[Token.StringLiteral] = 'String';
              TokenName[Token.RegularExpression] = 'RegularExpression';

              // A function following one of those tokens is an expression.
              FnExprTokens = ['(', '{', '[', 'in', 'typeof', 'instanceof', 'new', 'return', 'case', 'delete', 'throw', 'void',
              // assignment operators
              '=', '+=', '-=', '*=', '/=', '%=', '<<=', '>>=', '>>>=', '&=', '|=', '^=', ',',
              // binary/unary operators
              '+', '-', '*', '/', '%', '++', '--', '<<', '>>', '>>>', '&', '|', '^', '!', '~', '&&', '||', '?', ':', '===', '==', '>=', '<=', '<', '>', '!=', '!=='];
              Syntax = {
                AssignmentExpression: 'AssignmentExpression',
                ArrayExpression: 'ArrayExpression',
                BlockStatement: 'BlockStatement',
                BinaryExpression: 'BinaryExpression',
                BreakStatement: 'BreakStatement',
                CallExpression: 'CallExpression',
                CatchClause: 'CatchClause',
                ConditionalExpression: 'ConditionalExpression',
                ContinueStatement: 'ContinueStatement',
                DoWhileStatement: 'DoWhileStatement',
                DebuggerStatement: 'DebuggerStatement',
                EmptyStatement: 'EmptyStatement',
                ExpressionStatement: 'ExpressionStatement',
                ForStatement: 'ForStatement',
                ForInStatement: 'ForInStatement',
                FunctionDeclaration: 'FunctionDeclaration',
                FunctionExpression: 'FunctionExpression',
                Identifier: 'Identifier',
                IfStatement: 'IfStatement',
                Literal: 'Literal',
                LabeledStatement: 'LabeledStatement',
                LogicalExpression: 'LogicalExpression',
                MemberExpression: 'MemberExpression',
                NewExpression: 'NewExpression',
                ObjectExpression: 'ObjectExpression',
                Program: 'Program',
                Property: 'Property',
                ReturnStatement: 'ReturnStatement',
                SequenceExpression: 'SequenceExpression',
                SwitchStatement: 'SwitchStatement',
                SwitchCase: 'SwitchCase',
                ThisExpression: 'ThisExpression',
                ThrowStatement: 'ThrowStatement',
                TryStatement: 'TryStatement',
                UnaryExpression: 'UnaryExpression',
                UpdateExpression: 'UpdateExpression',
                VariableDeclaration: 'VariableDeclaration',
                VariableDeclarator: 'VariableDeclarator',
                WhileStatement: 'WhileStatement',
                WithStatement: 'WithStatement'
              };
              PropertyKind = {
                Data: 1,
                Get: 2,
                Set: 4
              };

              // Error messages should be identical to V8.
              Messages = {
                UnexpectedToken: 'Unexpected token %0',
                UnexpectedNumber: 'Unexpected number',
                UnexpectedString: 'Unexpected string',
                UnexpectedIdentifier: 'Unexpected identifier',
                UnexpectedReserved: 'Unexpected reserved word',
                UnexpectedEOS: 'Unexpected end of input',
                NewlineAfterThrow: 'Illegal newline after throw',
                InvalidRegExp: 'Invalid regular expression',
                UnterminatedRegExp: 'Invalid regular expression: missing /',
                InvalidLHSInAssignment: 'Invalid left-hand side in assignment',
                InvalidLHSInForIn: 'Invalid left-hand side in for-in',
                MultipleDefaultsInSwitch: 'More than one default clause in switch statement',
                NoCatchOrFinally: 'Missing catch or finally after try',
                UnknownLabel: 'Undefined label \'%0\'',
                Redeclaration: '%0 \'%1\' has already been declared',
                IllegalContinue: 'Illegal continue statement',
                IllegalBreak: 'Illegal break statement',
                IllegalReturn: 'Illegal return statement',
                StrictModeWith: 'Strict mode code may not include a with statement',
                StrictCatchVariable: 'Catch variable may not be eval or arguments in strict mode',
                StrictVarName: 'Variable name may not be eval or arguments in strict mode',
                StrictParamName: 'Parameter name eval or arguments is not allowed in strict mode',
                StrictParamDupe: 'Strict mode function may not have duplicate parameter names',
                StrictFunctionName: 'Function name may not be eval or arguments in strict mode',
                StrictOctalLiteral: 'Octal literals are not allowed in strict mode.',
                StrictDelete: 'Delete of an unqualified identifier in strict mode.',
                StrictDuplicateProperty: 'Duplicate data property in object literal not allowed in strict mode',
                AccessorDataProperty: 'Object literal may not have data and accessor property with the same name',
                AccessorGetSet: 'Object literal may not have multiple get/set accessors with the same name',
                StrictLHSAssignment: 'Assignment to eval or arguments is not allowed in strict mode',
                StrictLHSPostfix: 'Postfix increment/decrement may not have eval or arguments operand in strict mode',
                StrictLHSPrefix: 'Prefix increment/decrement may not have eval or arguments operand in strict mode',
                StrictReservedWord: 'Use of future reserved word in strict mode'
              };

              // See also tools/generate-unicode-regex.py.
              Regex = {
                NonAsciiIdentifierStart: new RegExp('[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0620-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u08A0\u08A2-\u08AC\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971-\u0977\u0979-\u097F\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D60\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1CF5\u1CF6\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005-\u3007\u3021-\u3029\u3031-\u3035\u3038-\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA66E\uA67F-\uA697\uA6A0-\uA6EF\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]'),
                NonAsciiIdentifierPart: new RegExp('[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0300-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u0483-\u0487\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u0591-\u05BD\u05BF\u05C1\u05C2\u05C4\u05C5\u05C7\u05D0-\u05EA\u05F0-\u05F2\u0610-\u061A\u0620-\u0669\u066E-\u06D3\u06D5-\u06DC\u06DF-\u06E8\u06EA-\u06FC\u06FF\u0710-\u074A\u074D-\u07B1\u07C0-\u07F5\u07FA\u0800-\u082D\u0840-\u085B\u08A0\u08A2-\u08AC\u08E4-\u08FE\u0900-\u0963\u0966-\u096F\u0971-\u0977\u0979-\u097F\u0981-\u0983\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BC-\u09C4\u09C7\u09C8\u09CB-\u09CE\u09D7\u09DC\u09DD\u09DF-\u09E3\u09E6-\u09F1\u0A01-\u0A03\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A3C\u0A3E-\u0A42\u0A47\u0A48\u0A4B-\u0A4D\u0A51\u0A59-\u0A5C\u0A5E\u0A66-\u0A75\u0A81-\u0A83\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABC-\u0AC5\u0AC7-\u0AC9\u0ACB-\u0ACD\u0AD0\u0AE0-\u0AE3\u0AE6-\u0AEF\u0B01-\u0B03\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3C-\u0B44\u0B47\u0B48\u0B4B-\u0B4D\u0B56\u0B57\u0B5C\u0B5D\u0B5F-\u0B63\u0B66-\u0B6F\u0B71\u0B82\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BBE-\u0BC2\u0BC6-\u0BC8\u0BCA-\u0BCD\u0BD0\u0BD7\u0BE6-\u0BEF\u0C01-\u0C03\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D-\u0C44\u0C46-\u0C48\u0C4A-\u0C4D\u0C55\u0C56\u0C58\u0C59\u0C60-\u0C63\u0C66-\u0C6F\u0C82\u0C83\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBC-\u0CC4\u0CC6-\u0CC8\u0CCA-\u0CCD\u0CD5\u0CD6\u0CDE\u0CE0-\u0CE3\u0CE6-\u0CEF\u0CF1\u0CF2\u0D02\u0D03\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D-\u0D44\u0D46-\u0D48\u0D4A-\u0D4E\u0D57\u0D60-\u0D63\u0D66-\u0D6F\u0D7A-\u0D7F\u0D82\u0D83\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0DCA\u0DCF-\u0DD4\u0DD6\u0DD8-\u0DDF\u0DF2\u0DF3\u0E01-\u0E3A\u0E40-\u0E4E\u0E50-\u0E59\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB9\u0EBB-\u0EBD\u0EC0-\u0EC4\u0EC6\u0EC8-\u0ECD\u0ED0-\u0ED9\u0EDC-\u0EDF\u0F00\u0F18\u0F19\u0F20-\u0F29\u0F35\u0F37\u0F39\u0F3E-\u0F47\u0F49-\u0F6C\u0F71-\u0F84\u0F86-\u0F97\u0F99-\u0FBC\u0FC6\u1000-\u1049\u1050-\u109D\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u135D-\u135F\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1714\u1720-\u1734\u1740-\u1753\u1760-\u176C\u176E-\u1770\u1772\u1773\u1780-\u17D3\u17D7\u17DC\u17DD\u17E0-\u17E9\u180B-\u180D\u1810-\u1819\u1820-\u1877\u1880-\u18AA\u18B0-\u18F5\u1900-\u191C\u1920-\u192B\u1930-\u193B\u1946-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u19D0-\u19D9\u1A00-\u1A1B\u1A20-\u1A5E\u1A60-\u1A7C\u1A7F-\u1A89\u1A90-\u1A99\u1AA7\u1B00-\u1B4B\u1B50-\u1B59\u1B6B-\u1B73\u1B80-\u1BF3\u1C00-\u1C37\u1C40-\u1C49\u1C4D-\u1C7D\u1CD0-\u1CD2\u1CD4-\u1CF6\u1D00-\u1DE6\u1DFC-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u200C\u200D\u203F\u2040\u2054\u2071\u207F\u2090-\u209C\u20D0-\u20DC\u20E1\u20E5-\u20F0\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D7F-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2DE0-\u2DFF\u2E2F\u3005-\u3007\u3021-\u302F\u3031-\u3035\u3038-\u303C\u3041-\u3096\u3099\u309A\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA62B\uA640-\uA66F\uA674-\uA67D\uA67F-\uA697\uA69F-\uA6F1\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA827\uA840-\uA873\uA880-\uA8C4\uA8D0-\uA8D9\uA8E0-\uA8F7\uA8FB\uA900-\uA92D\uA930-\uA953\uA960-\uA97C\uA980-\uA9C0\uA9CF-\uA9D9\uAA00-\uAA36\uAA40-\uAA4D\uAA50-\uAA59\uAA60-\uAA76\uAA7A\uAA7B\uAA80-\uAAC2\uAADB-\uAADD\uAAE0-\uAAEF\uAAF2-\uAAF6\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABEA\uABEC\uABED\uABF0-\uABF9\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE00-\uFE0F\uFE20-\uFE26\uFE33\uFE34\uFE4D-\uFE4F\uFE70-\uFE74\uFE76-\uFEFC\uFF10-\uFF19\uFF21-\uFF3A\uFF3F\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]')
              };

              // Ensure the condition is true, otherwise throw an error.
              // This is only to have a better contract semantic, i.e. another safety net
              // to catch a logic error. The condition shall be fulfilled in normal case.
              // Do NOT use this to enforce a certain condition on any user input.

              function assert(condition, message) {
                /* istanbul ignore if */
                if (!condition) {
                  throw new Error('ASSERT: ' + message);
                }
              }
              function isDecimalDigit(ch) {
                return ch >= 48 && ch <= 57; // 0..9
              }
              function isHexDigit(ch) {
                return '0123456789abcdefABCDEF'.indexOf(ch) >= 0;
              }
              function isOctalDigit(ch) {
                return '01234567'.indexOf(ch) >= 0;
              }

              // 7.2 White Space

              function isWhiteSpace(ch) {
                return ch === 0x20 || ch === 0x09 || ch === 0x0B || ch === 0x0C || ch === 0xA0 || ch >= 0x1680 && [0x1680, 0x180E, 0x2000, 0x2001, 0x2002, 0x2003, 0x2004, 0x2005, 0x2006, 0x2007, 0x2008, 0x2009, 0x200A, 0x202F, 0x205F, 0x3000, 0xFEFF].indexOf(ch) >= 0;
              }

              // 7.3 Line Terminators

              function isLineTerminator(ch) {
                return ch === 0x0A || ch === 0x0D || ch === 0x2028 || ch === 0x2029;
              }

              // 7.6 Identifier Names and Identifiers

              function isIdentifierStart(ch) {
                return ch == 0x40 || ch === 0x24 || ch === 0x5F ||
                // $ (dollar) and _ (underscore)
                ch >= 0x41 && ch <= 0x5A ||
                // A..Z
                ch >= 0x61 && ch <= 0x7A ||
                // a..z
                ch === 0x5C ||
                // \ (backslash)
                ch >= 0x80 && Regex.NonAsciiIdentifierStart.test(String.fromCharCode(ch));
              }
              function isIdentifierPart(ch) {
                return ch === 0x24 || ch === 0x5F ||
                // $ (dollar) and _ (underscore)
                ch >= 0x41 && ch <= 0x5A ||
                // A..Z
                ch >= 0x61 && ch <= 0x7A ||
                // a..z
                ch >= 0x30 && ch <= 0x39 ||
                // 0..9
                ch === 0x5C ||
                // \ (backslash)
                ch >= 0x80 && Regex.NonAsciiIdentifierPart.test(String.fromCharCode(ch));
              }

              // 7.6.1.2 Future Reserved Words

              function isFutureReservedWord(id) {
                switch (id) {
                  case 'class':
                  case 'enum':
                  case 'export':
                  case 'extends':
                  case 'import':
                  case 'super':
                    return true;
                  default:
                    return false;
                }
              }
              function isStrictModeReservedWord(id) {
                switch (id) {
                  case 'implements':
                  case 'interface':
                  case 'package':
                  case 'private':
                  case 'protected':
                  case 'public':
                  case 'static':
                  case 'yield':
                  case 'let':
                    return true;
                  default:
                    return false;
                }
              }
              function isRestrictedWord(id) {
                return id === 'eval' || id === 'arguments';
              }

              // 7.6.1.1 Keywords

              function isKeyword(id) {
                if (strict && isStrictModeReservedWord(id)) {
                  return true;
                }

                // 'const' is specialized as Keyword in V8.
                // 'yield' and 'let' are for compatiblity with SpiderMonkey and ES.next.
                // Some others are from future reserved words.

                switch (id.length) {
                  case 2:
                    return id === 'if' || id === 'in' || id === 'do';
                  case 3:
                    return id === 'var' || id === 'for' || id === 'new' || id === 'try' || id === 'let';
                  case 4:
                    return id === 'this' || id === 'else' || id === 'case' || id === 'void' || id === 'with' || id === 'enum';
                  case 5:
                    return id === 'while' || id === 'break' || id === 'catch' || id === 'throw' || id === 'const' || id === 'yield' || id === 'class' || id === 'super';
                  case 6:
                    return id === 'return' || id === 'typeof' || id === 'delete' || id === 'switch' || id === 'export' || id === 'import';
                  case 7:
                    return id === 'default' || id === 'finally' || id === 'extends';
                  case 8:
                    return id === 'function' || id === 'continue' || id === 'debugger';
                  case 10:
                    return id === 'instanceof';
                  default:
                    return false;
                }
              }

              // 7.4 Comments

              function addComment(type, value, start, end, loc) {
                var comment;
                assert(typeof start === 'number', 'Comment must have valid position');

                // Because the way the actual token is scanned, often the comments
                // (if any) are skipped twice during the lexical analysis.
                // Thus, we need to skip adding a comment if the comment array already
                // handled it.
                if (state.lastCommentStart >= start) {
                  return;
                }
                state.lastCommentStart = start;
                comment = {
                  type: type,
                  value: value
                };
                if (extra.range) {
                  comment.range = [start, end];
                }
                if (extra.loc) {
                  comment.loc = loc;
                }
                extra.comments.push(comment);
                if (extra.attachComment) {
                  extra.leadingComments.push(comment);
                  extra.trailingComments.push(comment);
                }
              }
              function skipSingleLineComment(offset) {
                var start, loc, ch, comment;
                start = index - offset;
                loc = {
                  start: {
                    line: lineNumber,
                    column: index - lineStart - offset
                  }
                };
                while (index < length) {
                  ch = source.charCodeAt(index);
                  ++index;
                  if (isLineTerminator(ch)) {
                    if (extra.comments) {
                      comment = source.slice(start + offset, index - 1);
                      loc.end = {
                        line: lineNumber,
                        column: index - lineStart - 1
                      };
                      addComment('Line', comment, start, index - 1, loc);
                    }
                    if (ch === 13 && source.charCodeAt(index) === 10) {
                      ++index;
                    }
                    ++lineNumber;
                    lineStart = index;
                    return;
                  }
                }
                if (extra.comments) {
                  comment = source.slice(start + offset, index);
                  loc.end = {
                    line: lineNumber,
                    column: index - lineStart
                  };
                  addComment('Line', comment, start, index, loc);
                }
              }
              function skipMultiLineComment() {
                var start, loc, ch, comment;
                if (extra.comments) {
                  start = index - 2;
                  loc = {
                    start: {
                      line: lineNumber,
                      column: index - lineStart - 2
                    }
                  };
                }
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (isLineTerminator(ch)) {
                    if (ch === 0x0D && source.charCodeAt(index + 1) === 0x0A) {
                      ++index;
                    }
                    ++lineNumber;
                    ++index;
                    lineStart = index;
                    if (index >= length) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                  } else if (ch === 0x2A) {
                    // Block comment ends with '*/'.
                    if (source.charCodeAt(index + 1) === 0x2F) {
                      ++index;
                      ++index;
                      if (extra.comments) {
                        comment = source.slice(start + 2, index - 2);
                        loc.end = {
                          line: lineNumber,
                          column: index - lineStart
                        };
                        addComment('Block', comment, start, index, loc);
                      }
                      return;
                    }
                    ++index;
                  } else {
                    ++index;
                  }
                }
                throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
              }
              function skipComment() {
                var ch, start;
                start = index === 0;
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (isWhiteSpace(ch)) {
                    ++index;
                  } else if (isLineTerminator(ch)) {
                    ++index;
                    if (ch === 0x0D && source.charCodeAt(index) === 0x0A) {
                      ++index;
                    }
                    ++lineNumber;
                    lineStart = index;
                    start = true;
                  } else if (ch === 0x2F) {
                    // U+002F is '/'
                    ch = source.charCodeAt(index + 1);
                    if (ch === 0x2F) {
                      ++index;
                      ++index;
                      skipSingleLineComment(2);
                      start = true;
                    } else if (ch === 0x2A) {
                      // U+002A is '*'
                      ++index;
                      ++index;
                      skipMultiLineComment();
                    } else {
                      break;
                    }
                  } else if (start && ch === 0x2D) {
                    // U+002D is '-'
                    // U+003E is '>'
                    if (source.charCodeAt(index + 1) === 0x2D && source.charCodeAt(index + 2) === 0x3E) {
                      // '-->' is a single-line comment
                      index += 3;
                      skipSingleLineComment(3);
                    } else {
                      break;
                    }
                  } else if (ch === 0x3C) {
                    // U+003C is '<'
                    if (source.slice(index + 1, index + 4) === '!--') {
                      ++index; // `<`
                      ++index; // `!`
                      ++index; // `-`
                      ++index; // `-`
                      skipSingleLineComment(4);
                    } else {
                      break;
                    }
                  } else {
                    break;
                  }
                }
              }
              function scanHexEscape(prefix) {
                var i,
                  len,
                  ch,
                  code = 0;
                len = prefix === 'u' ? 4 : 2;
                for (i = 0; i < len; ++i) {
                  if (index < length && isHexDigit(source[index])) {
                    ch = source[index++];
                    code = code * 16 + '0123456789abcdef'.indexOf(ch.toLowerCase());
                  } else {
                    return '';
                  }
                }
                return String.fromCharCode(code);
              }
              function getEscapedIdentifier() {
                var ch, id;
                ch = source.charCodeAt(index++);
                id = String.fromCharCode(ch);

                // '\u' (U+005C, U+0075) denotes an escaped character.
                if (ch === 0x5C) {
                  if (source.charCodeAt(index) !== 0x75) {
                    throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                  }
                  ++index;
                  ch = scanHexEscape('u');
                  if (!ch || ch === '\\' || !isIdentifierStart(ch.charCodeAt(0))) {
                    throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                  }
                  id = ch;
                }
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (!isIdentifierPart(ch)) {
                    break;
                  }
                  ++index;
                  id += String.fromCharCode(ch);

                  // '\u' (U+005C, U+0075) denotes an escaped character.
                  if (ch === 0x5C) {
                    id = id.substr(0, id.length - 1);
                    if (source.charCodeAt(index) !== 0x75) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                    ++index;
                    ch = scanHexEscape('u');
                    if (!ch || ch === '\\' || !isIdentifierPart(ch.charCodeAt(0))) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                    id += ch;
                  }
                }
                return id;
              }
              function getIdentifier() {
                var start, ch;
                start = index++;
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (ch === 0x5C) {
                    // Blackslash (U+005C) marks Unicode escape sequence.
                    index = start;
                    return getEscapedIdentifier();
                  }
                  if (isIdentifierPart(ch)) {
                    ++index;
                  } else {
                    break;
                  }
                }
                return source.slice(start, index);
              }
              function scanIdentifier() {
                var start, id, type;
                start = index;

                // Backslash (U+005C) starts an escaped character.
                id = source.charCodeAt(index) === 0x5C ? getEscapedIdentifier() : getIdentifier();

                // There is no keyword or literal with only one character.
                // Thus, it must be an identifier.
                if (id.length === 1) {
                  type = Token.Identifier;
                } else if (isKeyword(id)) {
                  type = Token.Keyword;
                } else if (id === 'null') {
                  type = Token.NullLiteral;
                } else if (id === 'true' || id === 'false') {
                  type = Token.BooleanLiteral;
                } else {
                  type = Token.Identifier;
                }
                return {
                  type: type,
                  value: id,
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }

              // 7.7 Punctuators

              function scanPunctuator() {
                var start = index,
                  code = source.charCodeAt(index),
                  code2,
                  ch1 = source[index],
                  ch2,
                  ch3,
                  ch4;
                switch (code) {
                  // Check for most common single-character punctuators.
                  case 0x2E: // . dot
                  case 0x28: // ( open bracket
                  case 0x29: // ) close bracket
                  case 0x3B: // ; semicolon
                  case 0x2C: // , comma
                  case 0x7B: // { open curly brace
                  case 0x7D: // } close curly brace
                  case 0x5B: // [
                  case 0x5D: // ]
                  case 0x3A: // :
                  case 0x3F: // ?
                  case 0x7E:
                    // ~
                    ++index;
                    if (extra.tokenize) {
                      if (code === 0x28) {
                        extra.openParenToken = extra.tokens.length;
                      } else if (code === 0x7B) {
                        extra.openCurlyToken = extra.tokens.length;
                      }
                    }
                    return {
                      type: Token.Punctuator,
                      value: String.fromCharCode(code),
                      lineNumber: lineNumber,
                      lineStart: lineStart,
                      start: start,
                      end: index
                    };
                  default:
                    code2 = source.charCodeAt(index + 1);

                    // '=' (U+003D) marks an assignment or comparison operator.
                    if (code2 === 0x3D) {
                      switch (code) {
                        case 0x2B: // +
                        case 0x2D: // -
                        case 0x2F: // /
                        case 0x3C: // <
                        case 0x3E: // >
                        case 0x5E: // ^
                        case 0x7C: // |
                        case 0x25: // %
                        case 0x26: // &
                        case 0x2A:
                          // *
                          index += 2;
                          return {
                            type: Token.Punctuator,
                            value: String.fromCharCode(code) + String.fromCharCode(code2),
                            lineNumber: lineNumber,
                            lineStart: lineStart,
                            start: start,
                            end: index
                          };
                        case 0x21: // !
                        case 0x3D:
                          // =
                          index += 2;

                          // !== and ===
                          if (source.charCodeAt(index) === 0x3D) {
                            ++index;
                          }
                          return {
                            type: Token.Punctuator,
                            value: source.slice(start, index),
                            lineNumber: lineNumber,
                            lineStart: lineStart,
                            start: start,
                            end: index
                          };
                      }
                    }
                }

                // 4-character punctuator: >>>=

                ch4 = source.substr(index, 4);
                if (ch4 === '>>>=') {
                  index += 4;
                  return {
                    type: Token.Punctuator,
                    value: ch4,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }

                // 3-character punctuators: === !== >>> <<= >>=

                ch3 = ch4.substr(0, 3);
                if (ch3 === '>>>' || ch3 === '<<=' || ch3 === '>>=') {
                  index += 3;
                  return {
                    type: Token.Punctuator,
                    value: ch3,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }

                // Other 2-character punctuators: ++ -- << >> && ||
                ch2 = ch3.substr(0, 2);
                if (ch1 === ch2[1] && '+-<>&|'.indexOf(ch1) >= 0 || ch2 === '=>') {
                  index += 2;
                  return {
                    type: Token.Punctuator,
                    value: ch2,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }

                // 1-character punctuators: < > = ! + - * % & | ^ /
                if ('<>=!+-*%&|^/'.indexOf(ch1) >= 0) {
                  ++index;
                  return {
                    type: Token.Punctuator,
                    value: ch1,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }
                throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
              }

              // 7.8.3 Numeric Literals

              function scanHexLiteral(start) {
                var number = '';
                while (index < length) {
                  if (!isHexDigit(source[index])) {
                    break;
                  }
                  number += source[index++];
                }
                if (number.length === 0) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                if (isIdentifierStart(source.charCodeAt(index))) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.NumericLiteral,
                  value: parseInt('0x' + number, 16),
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }
              function scanOctalLiteral(start) {
                var number = '0' + source[index++];
                while (index < length) {
                  if (!isOctalDigit(source[index])) {
                    break;
                  }
                  number += source[index++];
                }
                if (isIdentifierStart(source.charCodeAt(index)) || isDecimalDigit(source.charCodeAt(index))) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.NumericLiteral,
                  value: parseInt(number, 8),
                  octal: true,
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }
              function scanNumericLiteral() {
                var number, start, ch;
                ch = source[index];
                assert(isDecimalDigit(ch.charCodeAt(0)) || ch === '.', 'Numeric literal must start with a decimal digit or a decimal point');
                start = index;
                number = '';
                if (ch !== '.') {
                  number = source[index++];
                  ch = source[index];

                  // Hex number starts with '0x'.
                  // Octal number starts with '0'.
                  if (number === '0') {
                    if (ch === 'x' || ch === 'X') {
                      ++index;
                      return scanHexLiteral(start);
                    }
                    if (isOctalDigit(ch)) {
                      return scanOctalLiteral(start);
                    }

                    // decimal number starts with '0' such as '09' is illegal.
                    if (ch && isDecimalDigit(ch.charCodeAt(0))) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                  }
                  while (isDecimalDigit(source.charCodeAt(index))) {
                    number += source[index++];
                  }
                  ch = source[index];
                }
                if (ch === '.') {
                  number += source[index++];
                  while (isDecimalDigit(source.charCodeAt(index))) {
                    number += source[index++];
                  }
                  ch = source[index];
                }
                if (ch === 'e' || ch === 'E') {
                  number += source[index++];
                  ch = source[index];
                  if (ch === '+' || ch === '-') {
                    number += source[index++];
                  }
                  if (isDecimalDigit(source.charCodeAt(index))) {
                    while (isDecimalDigit(source.charCodeAt(index))) {
                      number += source[index++];
                    }
                  } else {
                    throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                  }
                }
                if (isIdentifierStart(source.charCodeAt(index))) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.NumericLiteral,
                  value: parseFloat(number),
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }

              // 7.8.4 String Literals

              function scanStringLiteral() {
                var str = '',
                  quote,
                  start,
                  ch,
                  code,
                  unescaped,
                  restore,
                  octal = false,
                  startLineNumber,
                  startLineStart;
                startLineNumber = lineNumber;
                startLineStart = lineStart;
                quote = source[index];
                assert(quote === '\'' || quote === '"', 'String literal must starts with a quote');
                start = index;
                ++index;
                while (index < length) {
                  ch = source[index++];
                  if (ch === quote) {
                    quote = '';
                    break;
                  } else if (ch === '\\') {
                    ch = source[index++];
                    if (!ch || !isLineTerminator(ch.charCodeAt(0))) {
                      switch (ch) {
                        case 'u':
                        case 'x':
                          restore = index;
                          unescaped = scanHexEscape(ch);
                          if (unescaped) {
                            str += unescaped;
                          } else {
                            index = restore;
                            str += ch;
                          }
                          break;
                        case 'n':
                          str += '\n';
                          break;
                        case 'r':
                          str += '\r';
                          break;
                        case 't':
                          str += '\t';
                          break;
                        case 'b':
                          str += '\b';
                          break;
                        case 'f':
                          str += '\f';
                          break;
                        case 'v':
                          str += '\x0B';
                          break;
                        default:
                          if (isOctalDigit(ch)) {
                            code = '01234567'.indexOf(ch);

                            // \0 is not octal escape sequence
                            if (code !== 0) {
                              octal = true;
                            }
                            if (index < length && isOctalDigit(source[index])) {
                              octal = true;
                              code = code * 8 + '01234567'.indexOf(source[index++]);

                              // 3 digits are only allowed when string starts
                              // with 0, 1, 2, 3
                              if ('0123'.indexOf(ch) >= 0 && index < length && isOctalDigit(source[index])) {
                                code = code * 8 + '01234567'.indexOf(source[index++]);
                              }
                            }
                            str += String.fromCharCode(code);
                          } else {
                            str += ch;
                          }
                          break;
                      }
                    } else {
                      ++lineNumber;
                      if (ch === '\r' && source[index] === '\n') {
                        ++index;
                      }
                      lineStart = index;
                    }
                  } else if (isLineTerminator(ch.charCodeAt(0))) {
                    break;
                  } else {
                    str += ch;
                  }
                }
                if (quote !== '') {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.StringLiteral,
                  value: str,
                  octal: octal,
                  startLineNumber: startLineNumber,
                  startLineStart: startLineStart,
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }
              function testRegExp(pattern, flags) {
                var value;
                try {
                  value = new RegExp(pattern, flags);
                } catch (e) {
                  throwError({}, Messages.InvalidRegExp);
                }
                return value;
              }
              function scanRegExpBody() {
                var ch, str, classMarker, terminated, body;
                ch = source[index];
                assert(ch === '/', 'Regular expression literal must start with a slash');
                str = source[index++];
                classMarker = false;
                terminated = false;
                while (index < length) {
                  ch = source[index++];
                  str += ch;
                  if (ch === '\\') {
                    ch = source[index++];
                    // ECMA-262 7.8.5
                    if (isLineTerminator(ch.charCodeAt(0))) {
                      throwError({}, Messages.UnterminatedRegExp);
                    }
                    str += ch;
                  } else if (isLineTerminator(ch.charCodeAt(0))) {
                    throwError({}, Messages.UnterminatedRegExp);
                  } else if (classMarker) {
                    if (ch === ']') {
                      classMarker = false;
                    }
                  } else {
                    if (ch === '/') {
                      terminated = true;
                      break;
                    } else if (ch === '[') {
                      classMarker = true;
                    }
                  }
                }
                if (!terminated) {
                  throwError({}, Messages.UnterminatedRegExp);
                }

                // Exclude leading and trailing slash.
                body = str.substr(1, str.length - 2);
                return {
                  value: body,
                  literal: str
                };
              }
              function scanRegExpFlags() {
                var ch, str, flags, restore;
                str = '';
                flags = '';
                while (index < length) {
                  ch = source[index];
                  if (!isIdentifierPart(ch.charCodeAt(0))) {
                    break;
                  }
                  ++index;
                  if (ch === '\\' && index < length) {
                    ch = source[index];
                    if (ch === 'u') {
                      ++index;
                      restore = index;
                      ch = scanHexEscape('u');
                      if (ch) {
                        flags += ch;
                        for (str += '\\u'; restore < index; ++restore) {
                          str += source[restore];
                        }
                      } else {
                        index = restore;
                        flags += 'u';
                        str += '\\u';
                      }
                      throwErrorTolerant({}, Messages.UnexpectedToken, 'ILLEGAL');
                    } else {
                      str += '\\';
                      throwErrorTolerant({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                  } else {
                    flags += ch;
                    str += ch;
                  }
                }
                return {
                  value: flags,
                  literal: str
                };
              }
              function scanRegExp() {
                var start, body, flags, value;
                lookahead = null;
                skipComment();
                start = index;
                body = scanRegExpBody();
                flags = scanRegExpFlags();
                value = testRegExp(body.value, flags.value);
                if (extra.tokenize) {
                  return {
                    type: Token.RegularExpression,
                    value: value,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }
                return {
                  literal: body.literal + flags.literal,
                  value: value,
                  start: start,
                  end: index
                };
              }
              function collectRegex() {
                var pos, loc, regex, token;
                skipComment();
                pos = index;
                loc = {
                  start: {
                    line: lineNumber,
                    column: index - lineStart
                  }
                };
                regex = scanRegExp();
                loc.end = {
                  line: lineNumber,
                  column: index - lineStart
                };

                /* istanbul ignore next */
                if (!extra.tokenize) {
                  // Pop the previous token, which is likely '/' or '/='
                  if (extra.tokens.length > 0) {
                    token = extra.tokens[extra.tokens.length - 1];
                    if (token.range[0] === pos && token.type === 'Punctuator') {
                      if (token.value === '/' || token.value === '/=') {
                        extra.tokens.pop();
                      }
                    }
                  }
                  extra.tokens.push({
                    type: 'RegularExpression',
                    value: regex.literal,
                    range: [pos, index],
                    loc: loc
                  });
                }
                return regex;
              }
              function isIdentifierName(token) {
                return token.type === Token.Identifier || token.type === Token.Keyword || token.type === Token.BooleanLiteral || token.type === Token.NullLiteral;
              }
              function advanceSlash() {
                var prevToken, checkToken;
                // Using the following algorithm:
                // https://github.com/mozilla/sweet.js/wiki/design
                prevToken = extra.tokens[extra.tokens.length - 1];
                if (!prevToken) {
                  // Nothing before that: it cannot be a division.
                  return collectRegex();
                }
                if (prevToken.type === 'Punctuator') {
                  if (prevToken.value === ']') {
                    return scanPunctuator();
                  }
                  if (prevToken.value === ')') {
                    checkToken = extra.tokens[extra.openParenToken - 1];
                    if (checkToken && checkToken.type === 'Keyword' && (checkToken.value === 'if' || checkToken.value === 'while' || checkToken.value === 'for' || checkToken.value === 'with')) {
                      return collectRegex();
                    }
                    return scanPunctuator();
                  }
                  if (prevToken.value === '}') {
                    // Dividing a function by anything makes little sense,
                    // but we have to check for that.
                    if (extra.tokens[extra.openCurlyToken - 3] && extra.tokens[extra.openCurlyToken - 3].type === 'Keyword') {
                      // Anonymous function.
                      checkToken = extra.tokens[extra.openCurlyToken - 4];
                      if (!checkToken) {
                        return scanPunctuator();
                      }
                    } else if (extra.tokens[extra.openCurlyToken - 4] && extra.tokens[extra.openCurlyToken - 4].type === 'Keyword') {
                      // Named function.
                      checkToken = extra.tokens[extra.openCurlyToken - 5];
                      if (!checkToken) {
                        return collectRegex();
                      }
                    } else {
                      return scanPunctuator();
                    }
                    // checkToken determines whether the function is
                    // a declaration or an expression.
                    if (FnExprTokens.indexOf(checkToken.value) >= 0) {
                      // It is an expression.
                      return scanPunctuator();
                    }
                    // It is a declaration.
                    return collectRegex();
                  }
                  return collectRegex();
                }
                if (prevToken.type === 'Keyword') {
                  return collectRegex();
                }
                return scanPunctuator();
              }
              function advance() {
                var ch;
                skipComment();
                if (index >= length) {
                  return {
                    type: Token.EOF,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: index,
                    end: index
                  };
                }
                ch = source.charCodeAt(index);
                if (isIdentifierStart(ch)) {
                  return scanIdentifier();
                }

                // Very common: ( and ) and ;
                if (ch === 0x28 || ch === 0x29 || ch === 0x3B) {
                  return scanPunctuator();
                }

                // String literal starts with single quote (U+0027) or double quote (U+0022).
                if (ch === 0x27 || ch === 0x22) {
                  return scanStringLiteral();
                }

                // Dot (.) U+002E can also start a floating-point number, hence the need
                // to check the next character.
                if (ch === 0x2E) {
                  if (isDecimalDigit(source.charCodeAt(index + 1))) {
                    return scanNumericLiteral();
                  }
                  return scanPunctuator();
                }
                if (isDecimalDigit(ch)) {
                  return scanNumericLiteral();
                }

                // Slash (/) U+002F can also start a regex.
                if (extra.tokenize && ch === 0x2F) {
                  return advanceSlash();
                }
                return scanPunctuator();
              }
              function collectToken() {
                var loc, token, value;
                skipComment();
                loc = {
                  start: {
                    line: lineNumber,
                    column: index - lineStart
                  }
                };
                token = advance();
                loc.end = {
                  line: lineNumber,
                  column: index - lineStart
                };
                if (token.type !== Token.EOF) {
                  value = source.slice(token.start, token.end);
                  extra.tokens.push({
                    type: TokenName[token.type],
                    value: value,
                    range: [token.start, token.end],
                    loc: loc
                  });
                }
                return token;
              }
              function lex() {
                var token;
                token = lookahead;
                index = token.end;
                lineNumber = token.lineNumber;
                lineStart = token.lineStart;
                lookahead = typeof extra.tokens !== 'undefined' ? collectToken() : advance();
                index = token.end;
                lineNumber = token.lineNumber;
                lineStart = token.lineStart;
                return token;
              }
              function peek() {
                var pos, line, start;
                pos = index;
                line = lineNumber;
                start = lineStart;
                lookahead = typeof extra.tokens !== 'undefined' ? collectToken() : advance();
                index = pos;
                lineNumber = line;
                lineStart = start;
              }
              function Position(line, column) {
                this.line = line;
                this.column = column;
              }
              function SourceLocation(startLine, startColumn, line, column) {
                this.start = new Position(startLine, startColumn);
                this.end = new Position(line, column);
              }
              SyntaxTreeDelegate = {
                name: 'SyntaxTree',
                processComment: function (node) {
                  var lastChild, trailingComments;
                  if (node.type === Syntax.Program) {
                    if (node.body.length > 0) {
                      return;
                    }
                  }
                  if (extra.trailingComments.length > 0) {
                    if (extra.trailingComments[0].range[0] >= node.range[1]) {
                      trailingComments = extra.trailingComments;
                      extra.trailingComments = [];
                    } else {
                      extra.trailingComments.length = 0;
                    }
                  } else {
                    if (extra.bottomRightStack.length > 0 && extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments && extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments[0].range[0] >= node.range[1]) {
                      trailingComments = extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments;
                      delete extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments;
                    }
                  }

                  // Eating the stack.
                  while (extra.bottomRightStack.length > 0 && extra.bottomRightStack[extra.bottomRightStack.length - 1].range[0] >= node.range[0]) {
                    lastChild = extra.bottomRightStack.pop();
                  }
                  if (lastChild) {
                    if (lastChild.leadingComments && lastChild.leadingComments[lastChild.leadingComments.length - 1].range[1] <= node.range[0]) {
                      node.leadingComments = lastChild.leadingComments;
                      delete lastChild.leadingComments;
                    }
                  } else if (extra.leadingComments.length > 0 && extra.leadingComments[extra.leadingComments.length - 1].range[1] <= node.range[0]) {
                    node.leadingComments = extra.leadingComments;
                    extra.leadingComments = [];
                  }
                  if (trailingComments) {
                    node.trailingComments = trailingComments;
                  }
                  extra.bottomRightStack.push(node);
                },
                markEnd: function (node, startToken) {
                  if (extra.range) {
                    node.range = [startToken.start, index];
                  }
                  if (extra.loc) {
                    node.loc = new SourceLocation(startToken.startLineNumber === undefined ? startToken.lineNumber : startToken.startLineNumber, startToken.start - (startToken.startLineStart === undefined ? startToken.lineStart : startToken.startLineStart), lineNumber, index - lineStart);
                    this.postProcess(node);
                  }
                  if (extra.attachComment) {
                    this.processComment(node);
                  }
                  return node;
                },
                postProcess: function (node) {
                  if (extra.source) {
                    node.loc.source = extra.source;
                  }
                  return node;
                },
                createArrayExpression: function (elements) {
                  return {
                    type: Syntax.ArrayExpression,
                    elements: elements
                  };
                },
                createAssignmentExpression: function (operator, left, right) {
                  return {
                    type: Syntax.AssignmentExpression,
                    operator: operator,
                    left: left,
                    right: right
                  };
                },
                createBinaryExpression: function (operator, left, right) {
                  var type = operator === '||' || operator === '&&' ? Syntax.LogicalExpression : Syntax.BinaryExpression;
                  return {
                    type: type,
                    operator: operator,
                    left: left,
                    right: right
                  };
                },
                createBlockStatement: function (body) {
                  return {
                    type: Syntax.BlockStatement,
                    body: body
                  };
                },
                createBreakStatement: function (label) {
                  return {
                    type: Syntax.BreakStatement,
                    label: label
                  };
                },
                createCallExpression: function (callee, args) {
                  return {
                    type: Syntax.CallExpression,
                    callee: callee,
                    'arguments': args
                  };
                },
                createCatchClause: function (param, body) {
                  return {
                    type: Syntax.CatchClause,
                    param: param,
                    body: body
                  };
                },
                createConditionalExpression: function (test, consequent, alternate) {
                  return {
                    type: Syntax.ConditionalExpression,
                    test: test,
                    consequent: consequent,
                    alternate: alternate
                  };
                },
                createContinueStatement: function (label) {
                  return {
                    type: Syntax.ContinueStatement,
                    label: label
                  };
                },
                createDebuggerStatement: function () {
                  return {
                    type: Syntax.DebuggerStatement
                  };
                },
                createDoWhileStatement: function (body, test) {
                  return {
                    type: Syntax.DoWhileStatement,
                    body: body,
                    test: test
                  };
                },
                createEmptyStatement: function () {
                  return {
                    type: Syntax.EmptyStatement
                  };
                },
                createExpressionStatement: function (expression) {
                  return {
                    type: Syntax.ExpressionStatement,
                    expression: expression
                  };
                },
                createForStatement: function (init, test, update, body) {
                  return {
                    type: Syntax.ForStatement,
                    init: init,
                    test: test,
                    update: update,
                    body: body
                  };
                },
                createForInStatement: function (left, right, body) {
                  return {
                    type: Syntax.ForInStatement,
                    left: left,
                    right: right,
                    body: body,
                    each: false
                  };
                },
                createFunctionDeclaration: function (id, params, defaults, body) {
                  return {
                    type: Syntax.FunctionDeclaration,
                    id: id,
                    params: params,
                    defaults: defaults,
                    body: body,
                    rest: null,
                    generator: false,
                    expression: false
                  };
                },
                createFunctionExpression: function (id, params, defaults, body) {
                  return {
                    type: Syntax.FunctionExpression,
                    id: id,
                    params: params,
                    defaults: defaults,
                    body: body,
                    rest: null,
                    generator: false,
                    expression: false
                  };
                },
                createIdentifier: function (name) {
                  return {
                    type: Syntax.Identifier,
                    name: name
                  };
                },
                createIfStatement: function (test, consequent, alternate) {
                  return {
                    type: Syntax.IfStatement,
                    test: test,
                    consequent: consequent,
                    alternate: alternate
                  };
                },
                createLabeledStatement: function (label, body) {
                  return {
                    type: Syntax.LabeledStatement,
                    label: label,
                    body: body
                  };
                },
                createLiteral: function (token) {
                  return {
                    type: Syntax.Literal,
                    value: token.value,
                    raw: source.slice(token.start, token.end)
                  };
                },
                createMemberExpression: function (accessor, object, property) {
                  return {
                    type: Syntax.MemberExpression,
                    computed: accessor === '[',
                    object: object,
                    property: property
                  };
                },
                createNewExpression: function (callee, args) {
                  return {
                    type: Syntax.NewExpression,
                    callee: callee,
                    'arguments': args
                  };
                },
                createObjectExpression: function (properties) {
                  return {
                    type: Syntax.ObjectExpression,
                    properties: properties
                  };
                },
                createPostfixExpression: function (operator, argument) {
                  return {
                    type: Syntax.UpdateExpression,
                    operator: operator,
                    argument: argument,
                    prefix: false
                  };
                },
                createProgram: function (body) {
                  return {
                    type: Syntax.Program,
                    body: body
                  };
                },
                createProperty: function (kind, key, value) {
                  return {
                    type: Syntax.Property,
                    key: key,
                    value: value,
                    kind: kind
                  };
                },
                createReturnStatement: function (argument) {
                  return {
                    type: Syntax.ReturnStatement,
                    argument: argument
                  };
                },
                createSequenceExpression: function (expressions) {
                  return {
                    type: Syntax.SequenceExpression,
                    expressions: expressions
                  };
                },
                createSwitchCase: function (test, consequent) {
                  return {
                    type: Syntax.SwitchCase,
                    test: test,
                    consequent: consequent
                  };
                },
                createSwitchStatement: function (discriminant, cases) {
                  return {
                    type: Syntax.SwitchStatement,
                    discriminant: discriminant,
                    cases: cases
                  };
                },
                createThisExpression: function () {
                  return {
                    type: Syntax.ThisExpression
                  };
                },
                createThrowStatement: function (argument) {
                  return {
                    type: Syntax.ThrowStatement,
                    argument: argument
                  };
                },
                createTryStatement: function (block, guardedHandlers, handlers, finalizer) {
                  return {
                    type: Syntax.TryStatement,
                    block: block,
                    guardedHandlers: guardedHandlers,
                    handlers: handlers,
                    finalizer: finalizer
                  };
                },
                createUnaryExpression: function (operator, argument) {
                  if (operator === '++' || operator === '--') {
                    return {
                      type: Syntax.UpdateExpression,
                      operator: operator,
                      argument: argument,
                      prefix: true
                    };
                  }
                  return {
                    type: Syntax.UnaryExpression,
                    operator: operator,
                    argument: argument,
                    prefix: true
                  };
                },
                createVariableDeclaration: function (declarations, kind) {
                  return {
                    type: Syntax.VariableDeclaration,
                    declarations: declarations,
                    kind: kind
                  };
                },
                createVariableDeclarator: function (id, init) {
                  return {
                    type: Syntax.VariableDeclarator,
                    id: id,
                    init: init
                  };
                },
                createWhileStatement: function (test, body) {
                  return {
                    type: Syntax.WhileStatement,
                    test: test,
                    body: body
                  };
                },
                createWithStatement: function (object, body) {
                  return {
                    type: Syntax.WithStatement,
                    object: object,
                    body: body
                  };
                }
              };

              // Return true if there is a line terminator before the next token.

              function peekLineTerminator() {
                var pos, line, start, found;
                pos = index;
                line = lineNumber;
                start = lineStart;
                skipComment();
                found = lineNumber !== line;
                index = pos;
                lineNumber = line;
                lineStart = start;
                return found;
              }

              // Throw an exception

              function throwError(token, messageFormat) {
                var error,
                  args = Array.prototype.slice.call(arguments, 2),
                  msg = messageFormat.replace(/%(\d)/g, function (whole, index) {
                    assert(index < args.length, 'Message reference must be in range');
                    return args[index];
                  });
                if (typeof token.lineNumber === 'number') {
                  error = new Error('Line ' + token.lineNumber + ': ' + msg);
                  error.index = token.start;
                  error.lineNumber = token.lineNumber;
                  error.column = token.start - lineStart + 1;
                } else {
                  error = new Error('Line ' + lineNumber + ': ' + msg);
                  error.index = index;
                  error.lineNumber = lineNumber;
                  error.column = index - lineStart + 1;
                }
                error.description = msg;
                throw error;
              }
              function throwErrorTolerant() {
                try {
                  throwError.apply(null, arguments);
                } catch (e) {
                  if (extra.errors) {
                    extra.errors.push(e);
                  } else {
                    throw e;
                  }
                }
              }

              // Throw an exception because of the token.

              function throwUnexpected(token) {
                if (token.type === Token.EOF) {
                  throwError(token, Messages.UnexpectedEOS);
                }
                if (token.type === Token.NumericLiteral) {
                  throwError(token, Messages.UnexpectedNumber);
                }
                if (token.type === Token.StringLiteral) {
                  throwError(token, Messages.UnexpectedString);
                }
                if (token.type === Token.Identifier) {
                  throwError(token, Messages.UnexpectedIdentifier);
                }
                if (token.type === Token.Keyword) {
                  if (isFutureReservedWord(token.value)) {
                    throwError(token, Messages.UnexpectedReserved);
                  } else if (strict && isStrictModeReservedWord(token.value)) {
                    throwErrorTolerant(token, Messages.StrictReservedWord);
                    return;
                  }
                  throwError(token, Messages.UnexpectedToken, token.value);
                }

                // BooleanLiteral, NullLiteral, or Punctuator.
                throwError(token, Messages.UnexpectedToken, token.value);
              }

              // Expect the next token to match the specified punctuator.
              // If not, an exception will be thrown.

              function expect(value) {
                var token = lex();
                if (token.type !== Token.Punctuator || token.value !== value) {
                  throwUnexpected(token);
                }
              }

              // Expect the next token to match the specified keyword.
              // If not, an exception will be thrown.

              function expectKeyword(keyword) {
                var token = lex();
                if (token.type !== Token.Keyword || token.value !== keyword) {
                  throwUnexpected(token);
                }
              }

              // Return true if the next token matches the specified punctuator.

              function match(value) {
                return lookahead.type === Token.Punctuator && lookahead.value === value;
              }

              // Return true if the next token matches the specified keyword

              function matchKeyword(keyword) {
                return lookahead.type === Token.Keyword && lookahead.value === keyword;
              }

              // Return true if the next token is an assignment operator

              function matchAssign() {
                var op;
                if (lookahead.type !== Token.Punctuator) {
                  return false;
                }
                op = lookahead.value;
                return op === '=' || op === '*=' || op === '/=' || op === '%=' || op === '+=' || op === '-=' || op === '<<=' || op === '>>=' || op === '>>>=' || op === '&=' || op === '^=' || op === '|=';
              }
              function consumeSemicolon() {
                var line;

                // Catch the very common case first: immediately a semicolon (U+003B).
                if (source.charCodeAt(index) === 0x3B || match(';')) {
                  lex();
                  return;
                }
                line = lineNumber;
                skipComment();
                if (lineNumber !== line) {
                  return;
                }
                if (lookahead.type !== Token.EOF && !match('}')) {
                  throwUnexpected(lookahead);
                }
              }

              // Return true if provided expression is LeftHandSideExpression

              function isLeftHandSide(expr) {
                return expr.type === Syntax.Identifier || expr.type === Syntax.MemberExpression;
              }

              // 11.1.4 Array Initialiser

              function parseArrayInitialiser() {
                var elements = [],
                  startToken;
                startToken = lookahead;
                expect('[');
                while (!match(']')) {
                  if (match(',')) {
                    lex();
                    elements.push(null);
                  } else {
                    elements.push(parseAssignmentExpression());
                    if (!match(']')) {
                      expect(',');
                    }
                  }
                }
                lex();
                return delegate.markEnd(delegate.createArrayExpression(elements), startToken);
              }

              // 11.1.5 Object Initialiser

              function parsePropertyFunction(param, first) {
                var previousStrict, body, startToken;
                previousStrict = strict;
                startToken = lookahead;
                body = parseFunctionSourceElements();
                if (first && strict && isRestrictedWord(param[0].name)) {
                  throwErrorTolerant(first, Messages.StrictParamName);
                }
                strict = previousStrict;
                return delegate.markEnd(delegate.createFunctionExpression(null, param, [], body), startToken);
              }
              function parseObjectPropertyKey() {
                var token, startToken;
                startToken = lookahead;
                token = lex();

                // Note: This function is called only from parseObjectProperty(), where
                // EOF and Punctuator tokens are already filtered out.

                if (token.type === Token.StringLiteral || token.type === Token.NumericLiteral) {
                  if (strict && token.octal) {
                    throwErrorTolerant(token, Messages.StrictOctalLiteral);
                  }
                  return delegate.markEnd(delegate.createLiteral(token), startToken);
                }
                return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
              }
              function parseObjectProperty() {
                var token, key, id, value, param, startToken;
                token = lookahead;
                startToken = lookahead;
                if (token.type === Token.Identifier) {
                  id = parseObjectPropertyKey();

                  // Property Assignment: Getter and Setter.

                  if (token.value === 'get' && !match(':')) {
                    key = parseObjectPropertyKey();
                    expect('(');
                    expect(')');
                    value = parsePropertyFunction([]);
                    return delegate.markEnd(delegate.createProperty('get', key, value), startToken);
                  }
                  if (token.value === 'set' && !match(':')) {
                    key = parseObjectPropertyKey();
                    expect('(');
                    token = lookahead;
                    if (token.type !== Token.Identifier) {
                      expect(')');
                      throwErrorTolerant(token, Messages.UnexpectedToken, token.value);
                      value = parsePropertyFunction([]);
                    } else {
                      param = [parseVariableIdentifier()];
                      expect(')');
                      value = parsePropertyFunction(param, token);
                    }
                    return delegate.markEnd(delegate.createProperty('set', key, value), startToken);
                  }
                  expect(':');
                  value = parseAssignmentExpression();
                  return delegate.markEnd(delegate.createProperty('init', id, value), startToken);
                }
                if (token.type === Token.EOF || token.type === Token.Punctuator) {
                  throwUnexpected(token);
                } else {
                  key = parseObjectPropertyKey();
                  expect(':');
                  value = parseAssignmentExpression();
                  return delegate.markEnd(delegate.createProperty('init', key, value), startToken);
                }
              }
              function parseObjectInitialiser() {
                var properties = [],
                  property,
                  name,
                  key,
                  kind,
                  map = {},
                  toString = String,
                  startToken;
                startToken = lookahead;
                expect('{');
                while (!match('}')) {
                  property = parseObjectProperty();
                  if (property.key.type === Syntax.Identifier) {
                    name = property.key.name;
                  } else {
                    name = toString(property.key.value);
                  }
                  kind = property.kind === 'init' ? PropertyKind.Data : property.kind === 'get' ? PropertyKind.Get : PropertyKind.Set;
                  key = '$' + name;
                  if (Object.prototype.hasOwnProperty.call(map, key)) {
                    if (map[key] === PropertyKind.Data) {
                      if (strict && kind === PropertyKind.Data) {
                        throwErrorTolerant({}, Messages.StrictDuplicateProperty);
                      } else if (kind !== PropertyKind.Data) {
                        throwErrorTolerant({}, Messages.AccessorDataProperty);
                      }
                    } else {
                      if (kind === PropertyKind.Data) {
                        throwErrorTolerant({}, Messages.AccessorDataProperty);
                      } else if (map[key] & kind) {
                        throwErrorTolerant({}, Messages.AccessorGetSet);
                      }
                    }
                    map[key] |= kind;
                  } else {
                    map[key] = kind;
                  }
                  properties.push(property);
                  if (!match('}')) {
                    expect(',');
                  }
                }
                expect('}');
                return delegate.markEnd(delegate.createObjectExpression(properties), startToken);
              }

              // 11.1.6 The Grouping Operator

              function parseGroupExpression() {
                var expr;
                expect('(');
                expr = parseExpression();
                expect(')');
                return expr;
              }

              // 11.1 Primary Expressions

              function parsePrimaryExpression() {
                var type, token, expr, startToken;
                if (match('(')) {
                  return parseGroupExpression();
                }
                if (match('[')) {
                  return parseArrayInitialiser();
                }
                if (match('{')) {
                  return parseObjectInitialiser();
                }
                type = lookahead.type;
                startToken = lookahead;
                if (type === Token.Identifier) {
                  expr = delegate.createIdentifier(lex().value);
                } else if (type === Token.StringLiteral || type === Token.NumericLiteral) {
                  if (strict && lookahead.octal) {
                    throwErrorTolerant(lookahead, Messages.StrictOctalLiteral);
                  }
                  expr = delegate.createLiteral(lex());
                } else if (type === Token.Keyword) {
                  if (matchKeyword('function')) {
                    return parseFunctionExpression();
                  }
                  if (matchKeyword('this')) {
                    lex();
                    expr = delegate.createThisExpression();
                  } else {
                    throwUnexpected(lex());
                  }
                } else if (type === Token.BooleanLiteral) {
                  token = lex();
                  token.value = token.value === 'true';
                  expr = delegate.createLiteral(token);
                } else if (type === Token.NullLiteral) {
                  token = lex();
                  token.value = null;
                  expr = delegate.createLiteral(token);
                } else if (match('/') || match('/=')) {
                  if (typeof extra.tokens !== 'undefined') {
                    expr = delegate.createLiteral(collectRegex());
                  } else {
                    expr = delegate.createLiteral(scanRegExp());
                  }
                  peek();
                } else {
                  throwUnexpected(lex());
                }
                return delegate.markEnd(expr, startToken);
              }

              // 11.2 Left-Hand-Side Expressions

              function parseArguments() {
                var args = [];
                expect('(');
                if (!match(')')) {
                  while (index < length) {
                    args.push(parseAssignmentExpression());
                    if (match(')')) {
                      break;
                    }
                    expect(',');
                  }
                }
                expect(')');
                return args;
              }
              function parseNonComputedProperty() {
                var token, startToken;
                startToken = lookahead;
                token = lex();
                if (!isIdentifierName(token)) {
                  throwUnexpected(token);
                }
                return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
              }
              function parseNonComputedMember() {
                expect('.');
                return parseNonComputedProperty();
              }
              function parseComputedMember() {
                var expr;
                expect('[');
                expr = parseExpression();
                expect(']');
                return expr;
              }
              function parseNewExpression() {
                var callee, args, startToken;
                startToken = lookahead;
                expectKeyword('new');
                callee = parseLeftHandSideExpression();
                args = match('(') ? parseArguments() : [];
                return delegate.markEnd(delegate.createNewExpression(callee, args), startToken);
              }
              function parseLeftHandSideExpressionAllowCall() {
                var previousAllowIn, expr, args, property, startToken;
                startToken = lookahead;
                previousAllowIn = state.allowIn;
                state.allowIn = true;
                expr = matchKeyword('new') ? parseNewExpression() : parsePrimaryExpression();
                state.allowIn = previousAllowIn;
                for (;;) {
                  if (match('.')) {
                    property = parseNonComputedMember();
                    expr = delegate.createMemberExpression('.', expr, property);
                  } else if (match('(')) {
                    args = parseArguments();
                    expr = delegate.createCallExpression(expr, args);
                  } else if (match('[')) {
                    property = parseComputedMember();
                    expr = delegate.createMemberExpression('[', expr, property);
                  } else {
                    break;
                  }
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }
              function parseLeftHandSideExpression() {
                var previousAllowIn, expr, property, startToken;
                startToken = lookahead;
                previousAllowIn = state.allowIn;
                expr = matchKeyword('new') ? parseNewExpression() : parsePrimaryExpression();
                state.allowIn = previousAllowIn;
                while (match('.') || match('[')) {
                  if (match('[')) {
                    property = parseComputedMember();
                    expr = delegate.createMemberExpression('[', expr, property);
                  } else {
                    property = parseNonComputedMember();
                    expr = delegate.createMemberExpression('.', expr, property);
                  }
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }

              // 11.3 Postfix Expressions

              function parsePostfixExpression() {
                var expr,
                  token,
                  startToken = lookahead;
                expr = parseLeftHandSideExpressionAllowCall();
                if (lookahead.type === Token.Punctuator) {
                  if ((match('++') || match('--')) && !peekLineTerminator()) {
                    // 11.3.1, 11.3.2
                    if (strict && expr.type === Syntax.Identifier && isRestrictedWord(expr.name)) {
                      throwErrorTolerant({}, Messages.StrictLHSPostfix);
                    }
                    if (!isLeftHandSide(expr)) {
                      throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
                    }
                    token = lex();
                    expr = delegate.markEnd(delegate.createPostfixExpression(token.value, expr), startToken);
                  }
                }
                return expr;
              }

              // 11.4 Unary Operators

              function parseUnaryExpression() {
                var token, expr, startToken;
                if (lookahead.type !== Token.Punctuator && lookahead.type !== Token.Keyword) {
                  expr = parsePostfixExpression();
                } else if (match('++') || match('--')) {
                  startToken = lookahead;
                  token = lex();
                  expr = parseUnaryExpression();
                  // 11.4.4, 11.4.5
                  if (strict && expr.type === Syntax.Identifier && isRestrictedWord(expr.name)) {
                    throwErrorTolerant({}, Messages.StrictLHSPrefix);
                  }
                  if (!isLeftHandSide(expr)) {
                    throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
                  }
                  expr = delegate.createUnaryExpression(token.value, expr);
                  expr = delegate.markEnd(expr, startToken);
                } else if (match('+') || match('-') || match('~') || match('!')) {
                  startToken = lookahead;
                  token = lex();
                  expr = parseUnaryExpression();
                  expr = delegate.createUnaryExpression(token.value, expr);
                  expr = delegate.markEnd(expr, startToken);
                } else if (matchKeyword('delete') || matchKeyword('void') || matchKeyword('typeof')) {
                  startToken = lookahead;
                  token = lex();
                  expr = parseUnaryExpression();
                  expr = delegate.createUnaryExpression(token.value, expr);
                  expr = delegate.markEnd(expr, startToken);
                  if (strict && expr.operator === 'delete' && expr.argument.type === Syntax.Identifier) {
                    throwErrorTolerant({}, Messages.StrictDelete);
                  }
                } else {
                  expr = parsePostfixExpression();
                }
                return expr;
              }
              function binaryPrecedence(token, allowIn) {
                var prec = 0;
                if (token.type !== Token.Punctuator && token.type !== Token.Keyword) {
                  return 0;
                }
                switch (token.value) {
                  case '||':
                    prec = 1;
                    break;
                  case '&&':
                    prec = 2;
                    break;
                  case '|':
                    prec = 3;
                    break;
                  case '^':
                    prec = 4;
                    break;
                  case '&':
                    prec = 5;
                    break;
                  case '==':
                  case '!=':
                  case '===':
                  case '!==':
                    prec = 6;
                    break;
                  case '<':
                  case '>':
                  case '<=':
                  case '>=':
                  case 'instanceof':
                    prec = 7;
                    break;
                  case 'in':
                    prec = allowIn ? 7 : 0;
                    break;
                  case '<<':
                  case '>>':
                  case '>>>':
                    prec = 8;
                    break;
                  case '+':
                  case '-':
                    prec = 9;
                    break;
                  case '*':
                  case '/':
                  case '%':
                    prec = 11;
                    break;
                }
                return prec;
              }

              // 11.5 Multiplicative Operators
              // 11.6 Additive Operators
              // 11.7 Bitwise Shift Operators
              // 11.8 Relational Operators
              // 11.9 Equality Operators
              // 11.10 Binary Bitwise Operators
              // 11.11 Binary Logical Operators

              function parseBinaryExpression() {
                var marker, markers, expr, token, prec, stack, right, operator, left, i;
                marker = lookahead;
                left = parseUnaryExpression();
                token = lookahead;
                prec = binaryPrecedence(token, state.allowIn);
                if (prec === 0) {
                  return left;
                }
                token.prec = prec;
                lex();
                markers = [marker, lookahead];
                right = parseUnaryExpression();
                stack = [left, token, right];
                while ((prec = binaryPrecedence(lookahead, state.allowIn)) > 0) {
                  // Reduce: make a binary expression from the three topmost entries.
                  while (stack.length > 2 && prec <= stack[stack.length - 2].prec) {
                    right = stack.pop();
                    operator = stack.pop().value;
                    left = stack.pop();
                    expr = delegate.createBinaryExpression(operator, left, right);
                    markers.pop();
                    marker = markers[markers.length - 1];
                    delegate.markEnd(expr, marker);
                    stack.push(expr);
                  }

                  // Shift.
                  token = lex();
                  token.prec = prec;
                  stack.push(token);
                  markers.push(lookahead);
                  expr = parseUnaryExpression();
                  stack.push(expr);
                }

                // Final reduce to clean-up the stack.
                i = stack.length - 1;
                expr = stack[i];
                markers.pop();
                while (i > 1) {
                  expr = delegate.createBinaryExpression(stack[i - 1].value, stack[i - 2], expr);
                  i -= 2;
                  marker = markers.pop();
                  delegate.markEnd(expr, marker);
                }
                return expr;
              }

              // 11.12 Conditional Operator

              function parseConditionalExpression() {
                var expr, previousAllowIn, consequent, alternate, startToken;
                startToken = lookahead;
                expr = parseBinaryExpression();
                if (match('?')) {
                  lex();
                  previousAllowIn = state.allowIn;
                  state.allowIn = true;
                  consequent = parseAssignmentExpression();
                  state.allowIn = previousAllowIn;
                  expect(':');
                  alternate = parseAssignmentExpression();
                  expr = delegate.createConditionalExpression(expr, consequent, alternate);
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }

              // 11.13 Assignment Operators

              function parseAssignmentExpression() {
                var token, left, right, node, startToken;
                token = lookahead;
                startToken = lookahead;
                node = left = parseConditionalExpression();
                if (matchAssign()) {
                  // LeftHandSideExpression
                  if (!isLeftHandSide(left)) {
                    throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
                  }

                  // 11.13.1
                  if (strict && left.type === Syntax.Identifier && isRestrictedWord(left.name)) {
                    throwErrorTolerant(token, Messages.StrictLHSAssignment);
                  }
                  token = lex();
                  right = parseAssignmentExpression();
                  node = delegate.markEnd(delegate.createAssignmentExpression(token.value, left, right), startToken);
                }
                return node;
              }

              // 11.14 Comma Operator

              function parseExpression() {
                var expr,
                  startToken = lookahead;
                expr = parseAssignmentExpression();
                if (match(',')) {
                  expr = delegate.createSequenceExpression([expr]);
                  while (index < length) {
                    if (!match(',')) {
                      break;
                    }
                    lex();
                    expr.expressions.push(parseAssignmentExpression());
                  }
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }

              // 12.1 Block

              function parseStatementList() {
                var list = [],
                  statement;
                while (index < length) {
                  if (match('}')) {
                    break;
                  }
                  statement = parseSourceElement();
                  if (typeof statement === 'undefined') {
                    break;
                  }
                  list.push(statement);
                }
                return list;
              }
              function parseBlock() {
                var block, startToken;
                startToken = lookahead;
                expect('{');
                block = parseStatementList();
                expect('}');
                return delegate.markEnd(delegate.createBlockStatement(block), startToken);
              }

              // 12.2 Variable Statement

              function parseVariableIdentifier() {
                var token, startToken;
                startToken = lookahead;
                token = lex();
                if (token.type !== Token.Identifier) {
                  throwUnexpected(token);
                }
                return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
              }
              function parseVariableDeclaration(kind) {
                var init = null,
                  id,
                  startToken;
                startToken = lookahead;
                id = parseVariableIdentifier();

                // 12.2.1
                if (strict && isRestrictedWord(id.name)) {
                  throwErrorTolerant({}, Messages.StrictVarName);
                }
                if (kind === 'const') {
                  expect('=');
                  init = parseAssignmentExpression();
                } else if (match('=')) {
                  lex();
                  init = parseAssignmentExpression();
                }
                return delegate.markEnd(delegate.createVariableDeclarator(id, init), startToken);
              }
              function parseVariableDeclarationList(kind) {
                var list = [];
                do {
                  list.push(parseVariableDeclaration(kind));
                  if (!match(',')) {
                    break;
                  }
                  lex();
                } while (index < length);
                return list;
              }
              function parseVariableStatement() {
                var declarations;
                expectKeyword('var');
                declarations = parseVariableDeclarationList();
                consumeSemicolon();
                return delegate.createVariableDeclaration(declarations, 'var');
              }

              // kind may be `const` or `let`
              // Both are experimental and not in the specification yet.
              // see http://wiki.ecmascript.org/doku.php?id=harmony:const
              // and http://wiki.ecmascript.org/doku.php?id=harmony:let
              function parseConstLetDeclaration(kind) {
                var declarations, startToken;
                startToken = lookahead;
                expectKeyword(kind);
                declarations = parseVariableDeclarationList(kind);
                consumeSemicolon();
                return delegate.markEnd(delegate.createVariableDeclaration(declarations, kind), startToken);
              }

              // 12.3 Empty Statement

              function parseEmptyStatement() {
                expect(';');
                return delegate.createEmptyStatement();
              }

              // 12.4 Expression Statement

              function parseExpressionStatement() {
                var expr = parseExpression();
                consumeSemicolon();
                return delegate.createExpressionStatement(expr);
              }

              // 12.5 If statement

              function parseIfStatement() {
                var test, consequent, alternate;
                expectKeyword('if');
                expect('(');
                test = parseExpression();
                expect(')');
                consequent = parseStatement();
                if (matchKeyword('else')) {
                  lex();
                  alternate = parseStatement();
                } else {
                  alternate = null;
                }
                return delegate.createIfStatement(test, consequent, alternate);
              }

              // 12.6 Iteration Statements

              function parseDoWhileStatement() {
                var body, test, oldInIteration;
                expectKeyword('do');
                oldInIteration = state.inIteration;
                state.inIteration = true;
                body = parseStatement();
                state.inIteration = oldInIteration;
                expectKeyword('while');
                expect('(');
                test = parseExpression();
                expect(')');
                if (match(';')) {
                  lex();
                }
                return delegate.createDoWhileStatement(body, test);
              }
              function parseWhileStatement() {
                var test, body, oldInIteration;
                expectKeyword('while');
                expect('(');
                test = parseExpression();
                expect(')');
                oldInIteration = state.inIteration;
                state.inIteration = true;
                body = parseStatement();
                state.inIteration = oldInIteration;
                return delegate.createWhileStatement(test, body);
              }
              function parseForVariableDeclaration() {
                var token, declarations, startToken;
                startToken = lookahead;
                token = lex();
                declarations = parseVariableDeclarationList();
                return delegate.markEnd(delegate.createVariableDeclaration(declarations, token.value), startToken);
              }
              function parseForStatement() {
                var init, test, update, left, right, body, oldInIteration;
                init = test = update = null;
                expectKeyword('for');
                expect('(');
                if (match(';')) {
                  lex();
                } else {
                  if (matchKeyword('var') || matchKeyword('let')) {
                    state.allowIn = false;
                    init = parseForVariableDeclaration();
                    state.allowIn = true;
                    if (init.declarations.length === 1 && matchKeyword('in')) {
                      lex();
                      left = init;
                      right = parseExpression();
                      init = null;
                    }
                  } else {
                    state.allowIn = false;
                    init = parseExpression();
                    state.allowIn = true;
                    if (matchKeyword('in')) {
                      // LeftHandSideExpression
                      if (!isLeftHandSide(init)) {
                        throwErrorTolerant({}, Messages.InvalidLHSInForIn);
                      }
                      lex();
                      left = init;
                      right = parseExpression();
                      init = null;
                    }
                  }
                  if (typeof left === 'undefined') {
                    expect(';');
                  }
                }
                if (typeof left === 'undefined') {
                  if (!match(';')) {
                    test = parseExpression();
                  }
                  expect(';');
                  if (!match(')')) {
                    update = parseExpression();
                  }
                }
                expect(')');
                oldInIteration = state.inIteration;
                state.inIteration = true;
                body = parseStatement();
                state.inIteration = oldInIteration;
                return typeof left === 'undefined' ? delegate.createForStatement(init, test, update, body) : delegate.createForInStatement(left, right, body);
              }

              // 12.7 The continue statement

              function parseContinueStatement() {
                var label = null,
                  key;
                expectKeyword('continue');

                // Optimize the most common form: 'continue;'.
                if (source.charCodeAt(index) === 0x3B) {
                  lex();
                  if (!state.inIteration) {
                    throwError({}, Messages.IllegalContinue);
                  }
                  return delegate.createContinueStatement(null);
                }
                if (peekLineTerminator()) {
                  if (!state.inIteration) {
                    throwError({}, Messages.IllegalContinue);
                  }
                  return delegate.createContinueStatement(null);
                }
                if (lookahead.type === Token.Identifier) {
                  label = parseVariableIdentifier();
                  key = '$' + label.name;
                  if (!Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
                    throwError({}, Messages.UnknownLabel, label.name);
                  }
                }
                consumeSemicolon();
                if (label === null && !state.inIteration) {
                  throwError({}, Messages.IllegalContinue);
                }
                return delegate.createContinueStatement(label);
              }

              // 12.8 The break statement

              function parseBreakStatement() {
                var label = null,
                  key;
                expectKeyword('break');

                // Catch the very common case first: immediately a semicolon (U+003B).
                if (source.charCodeAt(index) === 0x3B) {
                  lex();
                  if (!(state.inIteration || state.inSwitch)) {
                    throwError({}, Messages.IllegalBreak);
                  }
                  return delegate.createBreakStatement(null);
                }
                if (peekLineTerminator()) {
                  if (!(state.inIteration || state.inSwitch)) {
                    throwError({}, Messages.IllegalBreak);
                  }
                  return delegate.createBreakStatement(null);
                }
                if (lookahead.type === Token.Identifier) {
                  label = parseVariableIdentifier();
                  key = '$' + label.name;
                  if (!Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
                    throwError({}, Messages.UnknownLabel, label.name);
                  }
                }
                consumeSemicolon();
                if (label === null && !(state.inIteration || state.inSwitch)) {
                  throwError({}, Messages.IllegalBreak);
                }
                return delegate.createBreakStatement(label);
              }

              // 12.9 The return statement

              function parseReturnStatement() {
                var argument = null;
                expectKeyword('return');
                if (!state.inFunctionBody) {
                  throwErrorTolerant({}, Messages.IllegalReturn);
                }

                // 'return' followed by a space and an identifier is very common.
                if (source.charCodeAt(index) === 0x20) {
                  if (isIdentifierStart(source.charCodeAt(index + 1))) {
                    argument = parseExpression();
                    consumeSemicolon();
                    return delegate.createReturnStatement(argument);
                  }
                }
                if (peekLineTerminator()) {
                  return delegate.createReturnStatement(null);
                }
                if (!match(';')) {
                  if (!match('}') && lookahead.type !== Token.EOF) {
                    argument = parseExpression();
                  }
                }
                consumeSemicolon();
                return delegate.createReturnStatement(argument);
              }

              // 12.10 The with statement

              function parseWithStatement() {
                var object, body;
                if (strict) {
                  // TODO(ikarienator): Should we update the test cases instead?
                  skipComment();
                  throwErrorTolerant({}, Messages.StrictModeWith);
                }
                expectKeyword('with');
                expect('(');
                object = parseExpression();
                expect(')');
                body = parseStatement();
                return delegate.createWithStatement(object, body);
              }

              // 12.10 The swith statement

              function parseSwitchCase() {
                var test,
                  consequent = [],
                  statement,
                  startToken;
                startToken = lookahead;
                if (matchKeyword('default')) {
                  lex();
                  test = null;
                } else {
                  expectKeyword('case');
                  test = parseExpression();
                }
                expect(':');
                while (index < length) {
                  if (match('}') || matchKeyword('default') || matchKeyword('case')) {
                    break;
                  }
                  statement = parseStatement();
                  consequent.push(statement);
                }
                return delegate.markEnd(delegate.createSwitchCase(test, consequent), startToken);
              }
              function parseSwitchStatement() {
                var discriminant, cases, clause, oldInSwitch, defaultFound;
                expectKeyword('switch');
                expect('(');
                discriminant = parseExpression();
                expect(')');
                expect('{');
                cases = [];
                if (match('}')) {
                  lex();
                  return delegate.createSwitchStatement(discriminant, cases);
                }
                oldInSwitch = state.inSwitch;
                state.inSwitch = true;
                defaultFound = false;
                while (index < length) {
                  if (match('}')) {
                    break;
                  }
                  clause = parseSwitchCase();
                  if (clause.test === null) {
                    if (defaultFound) {
                      throwError({}, Messages.MultipleDefaultsInSwitch);
                    }
                    defaultFound = true;
                  }
                  cases.push(clause);
                }
                state.inSwitch = oldInSwitch;
                expect('}');
                return delegate.createSwitchStatement(discriminant, cases);
              }

              // 12.13 The throw statement

              function parseThrowStatement() {
                var argument;
                expectKeyword('throw');
                if (peekLineTerminator()) {
                  throwError({}, Messages.NewlineAfterThrow);
                }
                argument = parseExpression();
                consumeSemicolon();
                return delegate.createThrowStatement(argument);
              }

              // 12.14 The try statement

              function parseCatchClause() {
                var param, body, startToken;
                startToken = lookahead;
                expectKeyword('catch');
                expect('(');
                if (match(')')) {
                  throwUnexpected(lookahead);
                }
                param = parseVariableIdentifier();
                // 12.14.1
                if (strict && isRestrictedWord(param.name)) {
                  throwErrorTolerant({}, Messages.StrictCatchVariable);
                }
                expect(')');
                body = parseBlock();
                return delegate.markEnd(delegate.createCatchClause(param, body), startToken);
              }
              function parseTryStatement() {
                var block,
                  handlers = [],
                  finalizer = null;
                expectKeyword('try');
                block = parseBlock();
                if (matchKeyword('catch')) {
                  handlers.push(parseCatchClause());
                }
                if (matchKeyword('finally')) {
                  lex();
                  finalizer = parseBlock();
                }
                if (handlers.length === 0 && !finalizer) {
                  throwError({}, Messages.NoCatchOrFinally);
                }
                return delegate.createTryStatement(block, [], handlers, finalizer);
              }

              // 12.15 The debugger statement

              function parseDebuggerStatement() {
                expectKeyword('debugger');
                consumeSemicolon();
                return delegate.createDebuggerStatement();
              }

              // 12 Statements

              function parseStatement() {
                var type = lookahead.type,
                  expr,
                  labeledBody,
                  key,
                  startToken;
                if (type === Token.EOF) {
                  throwUnexpected(lookahead);
                }
                if (type === Token.Punctuator && lookahead.value === '{') {
                  return parseBlock();
                }
                startToken = lookahead;
                if (type === Token.Punctuator) {
                  switch (lookahead.value) {
                    case ';':
                      return delegate.markEnd(parseEmptyStatement(), startToken);
                    case '(':
                      return delegate.markEnd(parseExpressionStatement(), startToken);
                  }
                }
                if (type === Token.Keyword) {
                  switch (lookahead.value) {
                    case 'break':
                      return delegate.markEnd(parseBreakStatement(), startToken);
                    case 'continue':
                      return delegate.markEnd(parseContinueStatement(), startToken);
                    case 'debugger':
                      return delegate.markEnd(parseDebuggerStatement(), startToken);
                    case 'do':
                      return delegate.markEnd(parseDoWhileStatement(), startToken);
                    case 'for':
                      return delegate.markEnd(parseForStatement(), startToken);
                    case 'function':
                      return delegate.markEnd(parseFunctionDeclaration(), startToken);
                    case 'if':
                      return delegate.markEnd(parseIfStatement(), startToken);
                    case 'return':
                      return delegate.markEnd(parseReturnStatement(), startToken);
                    case 'switch':
                      return delegate.markEnd(parseSwitchStatement(), startToken);
                    case 'throw':
                      return delegate.markEnd(parseThrowStatement(), startToken);
                    case 'try':
                      return delegate.markEnd(parseTryStatement(), startToken);
                    case 'var':
                      return delegate.markEnd(parseVariableStatement(), startToken);
                    case 'while':
                      return delegate.markEnd(parseWhileStatement(), startToken);
                    case 'with':
                      return delegate.markEnd(parseWithStatement(), startToken);
                  }
                }
                expr = parseExpression();

                // 12.12 Labelled Statements
                if (expr.type === Syntax.Identifier && match(':')) {
                  lex();
                  key = '$' + expr.name;
                  if (Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
                    throwError({}, Messages.Redeclaration, 'Label', expr.name);
                  }
                  state.labelSet[key] = true;
                  labeledBody = parseStatement();
                  delete state.labelSet[key];
                  return delegate.markEnd(delegate.createLabeledStatement(expr, labeledBody), startToken);
                }
                consumeSemicolon();
                return delegate.markEnd(delegate.createExpressionStatement(expr), startToken);
              }

              // 13 Function Definition

              function parseFunctionSourceElements() {
                var sourceElement,
                  sourceElements = [],
                  token,
                  directive,
                  firstRestricted,
                  oldLabelSet,
                  oldInIteration,
                  oldInSwitch,
                  oldInFunctionBody,
                  startToken;
                startToken = lookahead;
                expect('{');
                while (index < length) {
                  if (lookahead.type !== Token.StringLiteral) {
                    break;
                  }
                  token = lookahead;
                  sourceElement = parseSourceElement();
                  sourceElements.push(sourceElement);
                  if (sourceElement.expression.type !== Syntax.Literal) {
                    // this is not directive
                    break;
                  }
                  directive = source.slice(token.start + 1, token.end - 1);
                  if (directive === 'use strict') {
                    strict = true;
                    if (firstRestricted) {
                      throwErrorTolerant(firstRestricted, Messages.StrictOctalLiteral);
                    }
                  } else {
                    if (!firstRestricted && token.octal) {
                      firstRestricted = token;
                    }
                  }
                }
                oldLabelSet = state.labelSet;
                oldInIteration = state.inIteration;
                oldInSwitch = state.inSwitch;
                oldInFunctionBody = state.inFunctionBody;
                state.labelSet = {};
                state.inIteration = false;
                state.inSwitch = false;
                state.inFunctionBody = true;
                while (index < length) {
                  if (match('}')) {
                    break;
                  }
                  sourceElement = parseSourceElement();
                  if (typeof sourceElement === 'undefined') {
                    break;
                  }
                  sourceElements.push(sourceElement);
                }
                expect('}');
                state.labelSet = oldLabelSet;
                state.inIteration = oldInIteration;
                state.inSwitch = oldInSwitch;
                state.inFunctionBody = oldInFunctionBody;
                return delegate.markEnd(delegate.createBlockStatement(sourceElements), startToken);
              }
              function parseParams(firstRestricted) {
                var param,
                  params = [],
                  token,
                  stricted,
                  paramSet,
                  key,
                  message;
                expect('(');
                if (!match(')')) {
                  paramSet = {};
                  while (index < length) {
                    token = lookahead;
                    param = parseVariableIdentifier();
                    key = '$' + token.value;
                    if (strict) {
                      if (isRestrictedWord(token.value)) {
                        stricted = token;
                        message = Messages.StrictParamName;
                      }
                      if (Object.prototype.hasOwnProperty.call(paramSet, key)) {
                        stricted = token;
                        message = Messages.StrictParamDupe;
                      }
                    } else if (!firstRestricted) {
                      if (isRestrictedWord(token.value)) {
                        firstRestricted = token;
                        message = Messages.StrictParamName;
                      } else if (isStrictModeReservedWord(token.value)) {
                        firstRestricted = token;
                        message = Messages.StrictReservedWord;
                      } else if (Object.prototype.hasOwnProperty.call(paramSet, key)) {
                        firstRestricted = token;
                        message = Messages.StrictParamDupe;
                      }
                    }
                    params.push(param);
                    paramSet[key] = true;
                    if (match(')')) {
                      break;
                    }
                    expect(',');
                  }
                }
                expect(')');
                return {
                  params: params,
                  stricted: stricted,
                  firstRestricted: firstRestricted,
                  message: message
                };
              }
              function parseFunctionDeclaration() {
                var id,
                  params = [],
                  body,
                  token,
                  stricted,
                  tmp,
                  firstRestricted,
                  message,
                  previousStrict,
                  startToken;
                startToken = lookahead;
                expectKeyword('function');
                token = lookahead;
                id = parseVariableIdentifier();
                if (strict) {
                  if (isRestrictedWord(token.value)) {
                    throwErrorTolerant(token, Messages.StrictFunctionName);
                  }
                } else {
                  if (isRestrictedWord(token.value)) {
                    firstRestricted = token;
                    message = Messages.StrictFunctionName;
                  } else if (isStrictModeReservedWord(token.value)) {
                    firstRestricted = token;
                    message = Messages.StrictReservedWord;
                  }
                }
                tmp = parseParams(firstRestricted);
                params = tmp.params;
                stricted = tmp.stricted;
                firstRestricted = tmp.firstRestricted;
                if (tmp.message) {
                  message = tmp.message;
                }
                previousStrict = strict;
                body = parseFunctionSourceElements();
                if (strict && firstRestricted) {
                  throwError(firstRestricted, message);
                }
                if (strict && stricted) {
                  throwErrorTolerant(stricted, message);
                }
                strict = previousStrict;
                return delegate.markEnd(delegate.createFunctionDeclaration(id, params, [], body), startToken);
              }
              function parseFunctionExpression() {
                var token,
                  id = null,
                  stricted,
                  firstRestricted,
                  message,
                  tmp,
                  params = [],
                  body,
                  previousStrict,
                  startToken;
                startToken = lookahead;
                expectKeyword('function');
                if (!match('(')) {
                  token = lookahead;
                  id = parseVariableIdentifier();
                  if (strict) {
                    if (isRestrictedWord(token.value)) {
                      throwErrorTolerant(token, Messages.StrictFunctionName);
                    }
                  } else {
                    if (isRestrictedWord(token.value)) {
                      firstRestricted = token;
                      message = Messages.StrictFunctionName;
                    } else if (isStrictModeReservedWord(token.value)) {
                      firstRestricted = token;
                      message = Messages.StrictReservedWord;
                    }
                  }
                }
                tmp = parseParams(firstRestricted);
                params = tmp.params;
                stricted = tmp.stricted;
                firstRestricted = tmp.firstRestricted;
                if (tmp.message) {
                  message = tmp.message;
                }
                previousStrict = strict;
                body = parseFunctionSourceElements();
                if (strict && firstRestricted) {
                  throwError(firstRestricted, message);
                }
                if (strict && stricted) {
                  throwErrorTolerant(stricted, message);
                }
                strict = previousStrict;
                return delegate.markEnd(delegate.createFunctionExpression(id, params, [], body), startToken);
              }

              // 14 Program

              function parseSourceElement() {
                if (lookahead.type === Token.Keyword) {
                  switch (lookahead.value) {
                    case 'const':
                    case 'let':
                      return parseConstLetDeclaration(lookahead.value);
                    case 'function':
                      return parseFunctionDeclaration();
                    default:
                      return parseStatement();
                  }
                }
                if (lookahead.type !== Token.EOF) {
                  return parseStatement();
                }
              }
              function parseSourceElements() {
                var sourceElement,
                  sourceElements = [],
                  token,
                  directive,
                  firstRestricted;
                while (index < length) {
                  token = lookahead;
                  if (token.type !== Token.StringLiteral) {
                    break;
                  }
                  sourceElement = parseSourceElement();
                  sourceElements.push(sourceElement);
                  if (sourceElement.expression.type !== Syntax.Literal) {
                    // this is not directive
                    break;
                  }
                  directive = source.slice(token.start + 1, token.end - 1);
                  if (directive === 'use strict') {
                    strict = true;
                    if (firstRestricted) {
                      throwErrorTolerant(firstRestricted, Messages.StrictOctalLiteral);
                    }
                  } else {
                    if (!firstRestricted && token.octal) {
                      firstRestricted = token;
                    }
                  }
                }
                while (index < length) {
                  sourceElement = parseSourceElement();
                  /* istanbul ignore if */
                  if (typeof sourceElement === 'undefined') {
                    break;
                  }
                  sourceElements.push(sourceElement);
                }
                return sourceElements;
              }
              function parseProgram() {
                var body, startToken;
                skipComment();
                peek();
                startToken = lookahead;
                strict = false;
                body = parseSourceElements();
                return delegate.markEnd(delegate.createProgram(body), startToken);
              }
              function filterTokenLocation() {
                var i,
                  entry,
                  token,
                  tokens = [];
                for (i = 0; i < extra.tokens.length; ++i) {
                  entry = extra.tokens[i];
                  token = {
                    type: entry.type,
                    value: entry.value
                  };
                  if (extra.range) {
                    token.range = entry.range;
                  }
                  if (extra.loc) {
                    token.loc = entry.loc;
                  }
                  tokens.push(token);
                }
                extra.tokens = tokens;
              }
              function tokenize(code, options) {
                var toString, token, tokens;
                toString = String;
                if (typeof code !== 'string' && !(code instanceof String)) {
                  code = toString(code);
                }
                delegate = SyntaxTreeDelegate;
                source = code;
                index = 0;
                lineNumber = source.length > 0 ? 1 : 0;
                lineStart = 0;
                length = source.length;
                lookahead = null;
                state = {
                  allowIn: true,
                  labelSet: {},
                  inFunctionBody: false,
                  inIteration: false,
                  inSwitch: false,
                  lastCommentStart: -1
                };
                extra = {};

                // Options matching.
                options = options || {};

                // Of course we collect tokens here.
                options.tokens = true;
                extra.tokens = [];
                extra.tokenize = true;
                // The following two fields are necessary to compute the Regex tokens.
                extra.openParenToken = -1;
                extra.openCurlyToken = -1;
                extra.range = typeof options.range === 'boolean' && options.range;
                extra.loc = typeof options.loc === 'boolean' && options.loc;
                if (typeof options.comment === 'boolean' && options.comment) {
                  extra.comments = [];
                }
                if (typeof options.tolerant === 'boolean' && options.tolerant) {
                  extra.errors = [];
                }
                try {
                  peek();
                  if (lookahead.type === Token.EOF) {
                    return extra.tokens;
                  }
                  token = lex();
                  while (lookahead.type !== Token.EOF) {
                    try {
                      token = lex();
                    } catch (lexError) {
                      token = lookahead;
                      if (extra.errors) {
                        extra.errors.push(lexError);
                        // We have to break on the first error
                        // to avoid infinite loops.
                        break;
                      } else {
                        throw lexError;
                      }
                    }
                  }
                  filterTokenLocation();
                  tokens = extra.tokens;
                  if (typeof extra.comments !== 'undefined') {
                    tokens.comments = extra.comments;
                  }
                  if (typeof extra.errors !== 'undefined') {
                    tokens.errors = extra.errors;
                  }
                } catch (e) {
                  throw e;
                } finally {
                  extra = {};
                }
                return tokens;
              }
              function parse(code, options) {
                var program, toString;
                toString = String;
                if (typeof code !== 'string' && !(code instanceof String)) {
                  code = toString(code);
                }
                delegate = SyntaxTreeDelegate;
                source = code;
                index = 0;
                lineNumber = source.length > 0 ? 1 : 0;
                lineStart = 0;
                length = source.length;
                lookahead = null;
                state = {
                  allowIn: true,
                  labelSet: {},
                  inFunctionBody: false,
                  inIteration: false,
                  inSwitch: false,
                  lastCommentStart: -1
                };
                extra = {};
                if (typeof options !== 'undefined') {
                  extra.range = typeof options.range === 'boolean' && options.range;
                  extra.loc = typeof options.loc === 'boolean' && options.loc;
                  extra.attachComment = typeof options.attachComment === 'boolean' && options.attachComment;
                  if (extra.loc && options.source !== null && options.source !== undefined) {
                    extra.source = toString(options.source);
                  }
                  if (typeof options.tokens === 'boolean' && options.tokens) {
                    extra.tokens = [];
                  }
                  if (typeof options.comment === 'boolean' && options.comment) {
                    extra.comments = [];
                  }
                  if (typeof options.tolerant === 'boolean' && options.tolerant) {
                    extra.errors = [];
                  }
                  if (extra.attachComment) {
                    extra.range = true;
                    extra.comments = [];
                    extra.bottomRightStack = [];
                    extra.trailingComments = [];
                    extra.leadingComments = [];
                  }
                }
                try {
                  program = parseProgram();
                  if (typeof extra.comments !== 'undefined') {
                    program.comments = extra.comments;
                  }
                  if (typeof extra.tokens !== 'undefined') {
                    filterTokenLocation();
                    program.tokens = extra.tokens;
                  }
                  if (typeof extra.errors !== 'undefined') {
                    program.errors = extra.errors;
                  }
                } catch (e) {
                  throw e;
                } finally {
                  extra = {};
                }
                return program;
              }

              // Sync with *.json manifests.
              exports.version = '1.2.2';
              exports.tokenize = tokenize;
              exports.parse = parse;

              // Deep copy.
              /* istanbul ignore next */
              exports.Syntax = function () {
                var name,
                  types = {};
                if (typeof Object.create === 'function') {
                  types = Object.create(null);
                }
                for (name in Syntax) {
                  if (Syntax.hasOwnProperty(name)) {
                    types[name] = Syntax[name];
                  }
                }
                if (typeof Object.freeze === 'function') {
                  Object.freeze(types);
                }
                return types;
              }();
            });
            /* vim: set sw=4 ts=4 et tw=80 : */
          }, {}],
          1: [function (require, module, exports) {
            (function (process) {
              /* parser generated by jison 0.4.13 */
              /*
                Returns a Parser object of the following structure:
              	  Parser: {
                  yy: {}
                }
              	  Parser.prototype: {
                  yy: {},
                  trace: function(),
                  symbols_: {associative list: name ==> number},
                  terminals_: {associative list: number ==> name},
                  productions_: [...],
                  performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate, $$, _$),
                  table: [...],
                  defaultActions: {...},
                  parseError: function(str, hash),
                  parse: function(input),
              	    lexer: {
                      EOF: 1,
                      parseError: function(str, hash),
                      setInput: function(input),
                      input: function(),
                      unput: function(str),
                      more: function(),
                      less: function(n),
                      pastInput: function(),
                      upcomingInput: function(),
                      showPosition: function(),
                      test_match: function(regex_match_array, rule_index),
                      next: function(),
                      lex: function(),
                      begin: function(condition),
                      popState: function(),
                      _currentRules: function(),
                      topState: function(),
                      pushState: function(condition),
              	        options: {
                          ranges: boolean           (optional: true ==> token location info will include a .range[] member)
                          flex: boolean             (optional: true ==> flex-like lexing behaviour where the rules are tested exhaustively to find the longest match)
                          backtrack_lexer: boolean  (optional: true ==> lexer regexes are tested in order and for each matching regex the action code is invoked; the lexer terminates the scan when a token is returned by the action code)
                      },
              	        performAction: function(yy, yy_, $avoiding_name_collisions, YY_START),
                      rules: [...],
                      conditions: {associative list: name ==> set},
                  }
                }
              
                token location info (@$, _$, etc.): {
                  first_line: n,
                  last_line: n,
                  first_column: n,
                  last_column: n,
                  range: [start_number, end_number]       (where the numbers are indexes into the input string, regular zero-based)
                }
              
                the parseError function receives a 'hash' object with these members for lexer and parser errors: {
                  text:        (matched text)
                  token:       (the produced terminal token, if any)
                  line:        (yylineno)
                }
                while parser (grammar) errors will also provide these members, i.e. parser errors deliver a superset of attributes: {
                  loc:         (yylloc)
                  expected:    (string describing the set of expected tokens)
                  recoverable: (boolean: TRUE when the parser has a error recovery rule available for this particular error)
                }
              */
              var parser = function () {
                var parser = {
                  trace: function trace() {},
                  yy: {},
                  symbols_: {
                    "error": 2,
                    "JSON_PATH": 3,
                    "DOLLAR": 4,
                    "PATH_COMPONENTS": 5,
                    "LEADING_CHILD_MEMBER_EXPRESSION": 6,
                    "PATH_COMPONENT": 7,
                    "MEMBER_COMPONENT": 8,
                    "SUBSCRIPT_COMPONENT": 9,
                    "CHILD_MEMBER_COMPONENT": 10,
                    "DESCENDANT_MEMBER_COMPONENT": 11,
                    "DOT": 12,
                    "MEMBER_EXPRESSION": 13,
                    "DOT_DOT": 14,
                    "STAR": 15,
                    "IDENTIFIER": 16,
                    "SCRIPT_EXPRESSION": 17,
                    "INTEGER": 18,
                    "END": 19,
                    "CHILD_SUBSCRIPT_COMPONENT": 20,
                    "DESCENDANT_SUBSCRIPT_COMPONENT": 21,
                    "[": 22,
                    "SUBSCRIPT": 23,
                    "]": 24,
                    "SUBSCRIPT_EXPRESSION": 25,
                    "SUBSCRIPT_EXPRESSION_LIST": 26,
                    "SUBSCRIPT_EXPRESSION_LISTABLE": 27,
                    ",": 28,
                    "STRING_LITERAL": 29,
                    "ARRAY_SLICE": 30,
                    "FILTER_EXPRESSION": 31,
                    "QQ_STRING": 32,
                    "Q_STRING": 33,
                    "$accept": 0,
                    "$end": 1
                  },
                  terminals_: {
                    2: "error",
                    4: "DOLLAR",
                    12: "DOT",
                    14: "DOT_DOT",
                    15: "STAR",
                    16: "IDENTIFIER",
                    17: "SCRIPT_EXPRESSION",
                    18: "INTEGER",
                    19: "END",
                    22: "[",
                    24: "]",
                    28: ",",
                    30: "ARRAY_SLICE",
                    31: "FILTER_EXPRESSION",
                    32: "QQ_STRING",
                    33: "Q_STRING"
                  },
                  productions_: [0, [3, 1], [3, 2], [3, 1], [3, 2], [5, 1], [5, 2], [7, 1], [7, 1], [8, 1], [8, 1], [10, 2], [6, 1], [11, 2], [13, 1], [13, 1], [13, 1], [13, 1], [13, 1], [9, 1], [9, 1], [20, 3], [21, 4], [23, 1], [23, 1], [26, 1], [26, 3], [27, 1], [27, 1], [27, 1], [25, 1], [25, 1], [25, 1], [29, 1], [29, 1]],
                  performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate /* action[1] */, $$ /* vstack */, _$ /* lstack */
                  /**/) {
                    /* this == yyval */
                    if (!yy.ast) {
                      yy.ast = _ast;
                      _ast.initialize();
                    }
                    var $0 = $$.length - 1;
                    switch (yystate) {
                      case 1:
                        yy.ast.set({
                          expression: {
                            type: "root",
                            value: $$[$0]
                          }
                        });
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 2:
                        yy.ast.set({
                          expression: {
                            type: "root",
                            value: $$[$0 - 1]
                          }
                        });
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 3:
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 4:
                        yy.ast.set({
                          operation: "member",
                          scope: "child",
                          expression: {
                            type: "identifier",
                            value: $$[$0 - 1]
                          }
                        });
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 5:
                        break;
                      case 6:
                        break;
                      case 7:
                        yy.ast.set({
                          operation: "member"
                        });
                        yy.ast.push();
                        break;
                      case 8:
                        yy.ast.set({
                          operation: "subscript"
                        });
                        yy.ast.push();
                        break;
                      case 9:
                        yy.ast.set({
                          scope: "child"
                        });
                        break;
                      case 10:
                        yy.ast.set({
                          scope: "descendant"
                        });
                        break;
                      case 11:
                        break;
                      case 12:
                        yy.ast.set({
                          scope: "child",
                          operation: "member"
                        });
                        break;
                      case 13:
                        break;
                      case 14:
                        yy.ast.set({
                          expression: {
                            type: "wildcard",
                            value: $$[$0]
                          }
                        });
                        break;
                      case 15:
                        yy.ast.set({
                          expression: {
                            type: "identifier",
                            value: $$[$0]
                          }
                        });
                        break;
                      case 16:
                        yy.ast.set({
                          expression: {
                            type: "script_expression",
                            value: $$[$0]
                          }
                        });
                        break;
                      case 17:
                        yy.ast.set({
                          expression: {
                            type: "numeric_literal",
                            value: parseInt($$[$0])
                          }
                        });
                        break;
                      case 18:
                        break;
                      case 19:
                        yy.ast.set({
                          scope: "child"
                        });
                        break;
                      case 20:
                        yy.ast.set({
                          scope: "descendant"
                        });
                        break;
                      case 21:
                        break;
                      case 22:
                        break;
                      case 23:
                        break;
                      case 24:
                        $$[$0].length > 1 ? yy.ast.set({
                          expression: {
                            type: "union",
                            value: $$[$0]
                          }
                        }) : this.$ = $$[$0];
                        break;
                      case 25:
                        this.$ = [$$[$0]];
                        break;
                      case 26:
                        this.$ = $$[$0 - 2].concat($$[$0]);
                        break;
                      case 27:
                        this.$ = {
                          expression: {
                            type: "numeric_literal",
                            value: parseInt($$[$0])
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 28:
                        this.$ = {
                          expression: {
                            type: "string_literal",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 29:
                        this.$ = {
                          expression: {
                            type: "slice",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 30:
                        this.$ = {
                          expression: {
                            type: "wildcard",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 31:
                        this.$ = {
                          expression: {
                            type: "script_expression",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 32:
                        this.$ = {
                          expression: {
                            type: "filter_expression",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 33:
                        this.$ = $$[$0];
                        break;
                      case 34:
                        this.$ = $$[$0];
                        break;
                    }
                  },
                  table: [{
                    3: 1,
                    4: [1, 2],
                    6: 3,
                    13: 4,
                    15: [1, 5],
                    16: [1, 6],
                    17: [1, 7],
                    18: [1, 8],
                    19: [1, 9]
                  }, {
                    1: [3]
                  }, {
                    1: [2, 1],
                    5: 10,
                    7: 11,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 3],
                    5: 21,
                    7: 11,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 12],
                    12: [2, 12],
                    14: [2, 12],
                    22: [2, 12]
                  }, {
                    1: [2, 14],
                    12: [2, 14],
                    14: [2, 14],
                    22: [2, 14]
                  }, {
                    1: [2, 15],
                    12: [2, 15],
                    14: [2, 15],
                    22: [2, 15]
                  }, {
                    1: [2, 16],
                    12: [2, 16],
                    14: [2, 16],
                    22: [2, 16]
                  }, {
                    1: [2, 17],
                    12: [2, 17],
                    14: [2, 17],
                    22: [2, 17]
                  }, {
                    1: [2, 18],
                    12: [2, 18],
                    14: [2, 18],
                    22: [2, 18]
                  }, {
                    1: [2, 2],
                    7: 22,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 5],
                    12: [2, 5],
                    14: [2, 5],
                    22: [2, 5]
                  }, {
                    1: [2, 7],
                    12: [2, 7],
                    14: [2, 7],
                    22: [2, 7]
                  }, {
                    1: [2, 8],
                    12: [2, 8],
                    14: [2, 8],
                    22: [2, 8]
                  }, {
                    1: [2, 9],
                    12: [2, 9],
                    14: [2, 9],
                    22: [2, 9]
                  }, {
                    1: [2, 10],
                    12: [2, 10],
                    14: [2, 10],
                    22: [2, 10]
                  }, {
                    1: [2, 19],
                    12: [2, 19],
                    14: [2, 19],
                    22: [2, 19]
                  }, {
                    1: [2, 20],
                    12: [2, 20],
                    14: [2, 20],
                    22: [2, 20]
                  }, {
                    13: 23,
                    15: [1, 5],
                    16: [1, 6],
                    17: [1, 7],
                    18: [1, 8],
                    19: [1, 9]
                  }, {
                    13: 24,
                    15: [1, 5],
                    16: [1, 6],
                    17: [1, 7],
                    18: [1, 8],
                    19: [1, 9],
                    22: [1, 25]
                  }, {
                    15: [1, 29],
                    17: [1, 30],
                    18: [1, 33],
                    23: 26,
                    25: 27,
                    26: 28,
                    27: 32,
                    29: 34,
                    30: [1, 35],
                    31: [1, 31],
                    32: [1, 36],
                    33: [1, 37]
                  }, {
                    1: [2, 4],
                    7: 22,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 6],
                    12: [2, 6],
                    14: [2, 6],
                    22: [2, 6]
                  }, {
                    1: [2, 11],
                    12: [2, 11],
                    14: [2, 11],
                    22: [2, 11]
                  }, {
                    1: [2, 13],
                    12: [2, 13],
                    14: [2, 13],
                    22: [2, 13]
                  }, {
                    15: [1, 29],
                    17: [1, 30],
                    18: [1, 33],
                    23: 38,
                    25: 27,
                    26: 28,
                    27: 32,
                    29: 34,
                    30: [1, 35],
                    31: [1, 31],
                    32: [1, 36],
                    33: [1, 37]
                  }, {
                    24: [1, 39]
                  }, {
                    24: [2, 23]
                  }, {
                    24: [2, 24],
                    28: [1, 40]
                  }, {
                    24: [2, 30]
                  }, {
                    24: [2, 31]
                  }, {
                    24: [2, 32]
                  }, {
                    24: [2, 25],
                    28: [2, 25]
                  }, {
                    24: [2, 27],
                    28: [2, 27]
                  }, {
                    24: [2, 28],
                    28: [2, 28]
                  }, {
                    24: [2, 29],
                    28: [2, 29]
                  }, {
                    24: [2, 33],
                    28: [2, 33]
                  }, {
                    24: [2, 34],
                    28: [2, 34]
                  }, {
                    24: [1, 41]
                  }, {
                    1: [2, 21],
                    12: [2, 21],
                    14: [2, 21],
                    22: [2, 21]
                  }, {
                    18: [1, 33],
                    27: 42,
                    29: 34,
                    30: [1, 35],
                    32: [1, 36],
                    33: [1, 37]
                  }, {
                    1: [2, 22],
                    12: [2, 22],
                    14: [2, 22],
                    22: [2, 22]
                  }, {
                    24: [2, 26],
                    28: [2, 26]
                  }],
                  defaultActions: {
                    27: [2, 23],
                    29: [2, 30],
                    30: [2, 31],
                    31: [2, 32]
                  },
                  parseError: function parseError(str, hash) {
                    if (hash.recoverable) {
                      this.trace(str);
                    } else {
                      throw new Error(str);
                    }
                  },
                  parse: function parse(input) {
                    var self = this,
                      stack = [0],
                      vstack = [null],
                      lstack = [],
                      table = this.table,
                      yytext = '',
                      yylineno = 0,
                      yyleng = 0,
                      TERROR = 2,
                      EOF = 1;
                    var args = lstack.slice.call(arguments, 1);
                    this.lexer.setInput(input);
                    this.lexer.yy = this.yy;
                    this.yy.lexer = this.lexer;
                    this.yy.parser = this;
                    if (typeof this.lexer.yylloc == 'undefined') {
                      this.lexer.yylloc = {};
                    }
                    var yyloc = this.lexer.yylloc;
                    lstack.push(yyloc);
                    var ranges = this.lexer.options && this.lexer.options.ranges;
                    if (typeof this.yy.parseError === 'function') {
                      this.parseError = this.yy.parseError;
                    } else {
                      this.parseError = Object.getPrototypeOf(this).parseError;
                    }
                    function lex() {
                      var token;
                      token = self.lexer.lex() || EOF;
                      if (typeof token !== 'number') {
                        token = self.symbols_[token] || token;
                      }
                      return token;
                    }
                    var symbol,
                      state,
                      action,
                      r,
                      yyval = {},
                      p,
                      len,
                      newState,
                      expected;
                    while (true) {
                      state = stack[stack.length - 1];
                      if (this.defaultActions[state]) {
                        action = this.defaultActions[state];
                      } else {
                        if (symbol === null || typeof symbol == 'undefined') {
                          symbol = lex();
                        }
                        action = table[state] && table[state][symbol];
                      }
                      if (typeof action === 'undefined' || !action.length || !action[0]) {
                        var errStr = '';
                        expected = [];
                        for (p in table[state]) {
                          if (this.terminals_[p] && p > TERROR) {
                            expected.push('\'' + this.terminals_[p] + '\'');
                          }
                        }
                        if (this.lexer.showPosition) {
                          errStr = 'Parse error on line ' + (yylineno + 1) + ':\n' + this.lexer.showPosition() + '\nExpecting ' + expected.join(', ') + ', got \'' + (this.terminals_[symbol] || symbol) + '\'';
                        } else {
                          errStr = 'Parse error on line ' + (yylineno + 1) + ': Unexpected ' + (symbol == EOF ? 'end of input' : '\'' + (this.terminals_[symbol] || symbol) + '\'');
                        }
                        this.parseError(errStr, {
                          text: this.lexer.match,
                          token: this.terminals_[symbol] || symbol,
                          line: this.lexer.yylineno,
                          loc: yyloc,
                          expected: expected
                        });
                      }
                      if (action[0] instanceof Array && action.length > 1) {
                        throw new Error('Parse Error: multiple actions possible at state: ' + state + ', token: ' + symbol);
                      }
                      switch (action[0]) {
                        case 1:
                          stack.push(symbol);
                          vstack.push(this.lexer.yytext);
                          lstack.push(this.lexer.yylloc);
                          stack.push(action[1]);
                          symbol = null;
                          {
                            yyleng = this.lexer.yyleng;
                            yytext = this.lexer.yytext;
                            yylineno = this.lexer.yylineno;
                            yyloc = this.lexer.yylloc;
                          }
                          break;
                        case 2:
                          len = this.productions_[action[1]][1];
                          yyval.$ = vstack[vstack.length - len];
                          yyval._$ = {
                            first_line: lstack[lstack.length - (len || 1)].first_line,
                            last_line: lstack[lstack.length - 1].last_line,
                            first_column: lstack[lstack.length - (len || 1)].first_column,
                            last_column: lstack[lstack.length - 1].last_column
                          };
                          if (ranges) {
                            yyval._$.range = [lstack[lstack.length - (len || 1)].range[0], lstack[lstack.length - 1].range[1]];
                          }
                          r = this.performAction.apply(yyval, [yytext, yyleng, yylineno, this.yy, action[1], vstack, lstack].concat(args));
                          if (typeof r !== 'undefined') {
                            return r;
                          }
                          if (len) {
                            stack = stack.slice(0, -1 * len * 2);
                            vstack = vstack.slice(0, -1 * len);
                            lstack = lstack.slice(0, -1 * len);
                          }
                          stack.push(this.productions_[action[1]][0]);
                          vstack.push(yyval.$);
                          lstack.push(yyval._$);
                          newState = table[stack[stack.length - 2]][stack[stack.length - 1]];
                          stack.push(newState);
                          break;
                        case 3:
                          return true;
                      }
                    }
                    return true;
                  }
                };
                var _ast = {
                  initialize: function () {
                    this._nodes = [];
                    this._node = {};
                    this._stash = [];
                  },
                  set: function (props) {
                    for (var k in props) this._node[k] = props[k];
                    return this._node;
                  },
                  node: function (obj) {
                    if (arguments.length) this._node = obj;
                    return this._node;
                  },
                  push: function () {
                    this._nodes.push(this._node);
                    this._node = {};
                  },
                  unshift: function () {
                    this._nodes.unshift(this._node);
                    this._node = {};
                  },
                  yield: function () {
                    var _nodes = this._nodes;
                    this.initialize();
                    return _nodes;
                  }
                };
                /* generated by jison-lex 0.2.1 */
                var lexer = function () {
                  var lexer = {
                    EOF: 1,
                    parseError: function parseError(str, hash) {
                      if (this.yy.parser) {
                        this.yy.parser.parseError(str, hash);
                      } else {
                        throw new Error(str);
                      }
                    },
                    // resets the lexer, sets new input
                    setInput: function (input) {
                      this._input = input;
                      this._more = this._backtrack = this.done = false;
                      this.yylineno = this.yyleng = 0;
                      this.yytext = this.matched = this.match = '';
                      this.conditionStack = ['INITIAL'];
                      this.yylloc = {
                        first_line: 1,
                        first_column: 0,
                        last_line: 1,
                        last_column: 0
                      };
                      if (this.options.ranges) {
                        this.yylloc.range = [0, 0];
                      }
                      this.offset = 0;
                      return this;
                    },
                    // consumes and returns one char from the input
                    input: function () {
                      var ch = this._input[0];
                      this.yytext += ch;
                      this.yyleng++;
                      this.offset++;
                      this.match += ch;
                      this.matched += ch;
                      var lines = ch.match(/(?:\r\n?|\n).*/g);
                      if (lines) {
                        this.yylineno++;
                        this.yylloc.last_line++;
                      } else {
                        this.yylloc.last_column++;
                      }
                      if (this.options.ranges) {
                        this.yylloc.range[1]++;
                      }
                      this._input = this._input.slice(1);
                      return ch;
                    },
                    // unshifts one char (or a string) into the input
                    unput: function (ch) {
                      var len = ch.length;
                      var lines = ch.split(/(?:\r\n?|\n)/g);
                      this._input = ch + this._input;
                      this.yytext = this.yytext.substr(0, this.yytext.length - len - 1);
                      //this.yyleng -= len;
                      this.offset -= len;
                      var oldLines = this.match.split(/(?:\r\n?|\n)/g);
                      this.match = this.match.substr(0, this.match.length - 1);
                      this.matched = this.matched.substr(0, this.matched.length - 1);
                      if (lines.length - 1) {
                        this.yylineno -= lines.length - 1;
                      }
                      var r = this.yylloc.range;
                      this.yylloc = {
                        first_line: this.yylloc.first_line,
                        last_line: this.yylineno + 1,
                        first_column: this.yylloc.first_column,
                        last_column: lines ? (lines.length === oldLines.length ? this.yylloc.first_column : 0) + oldLines[oldLines.length - lines.length].length - lines[0].length : this.yylloc.first_column - len
                      };
                      if (this.options.ranges) {
                        this.yylloc.range = [r[0], r[0] + this.yyleng - len];
                      }
                      this.yyleng = this.yytext.length;
                      return this;
                    },
                    // When called from action, caches matched text and appends it on next action
                    more: function () {
                      this._more = true;
                      return this;
                    },
                    // When called from action, signals the lexer that this rule fails to match the input, so the next matching rule (regex) should be tested instead.
                    reject: function () {
                      if (this.options.backtrack_lexer) {
                        this._backtrack = true;
                      } else {
                        return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. You can only invoke reject() in the lexer when the lexer is of the backtracking persuasion (options.backtrack_lexer = true).\n' + this.showPosition(), {
                          text: "",
                          token: null,
                          line: this.yylineno
                        });
                      }
                      return this;
                    },
                    // retain first n characters of the match
                    less: function (n) {
                      this.unput(this.match.slice(n));
                    },
                    // displays already matched input, i.e. for error messages
                    pastInput: function () {
                      var past = this.matched.substr(0, this.matched.length - this.match.length);
                      return (past.length > 20 ? '...' : '') + past.substr(-20).replace(/\n/g, "");
                    },
                    // displays upcoming input, i.e. for error messages
                    upcomingInput: function () {
                      var next = this.match;
                      if (next.length < 20) {
                        next += this._input.substr(0, 20 - next.length);
                      }
                      return (next.substr(0, 20) + (next.length > 20 ? '...' : '')).replace(/\n/g, "");
                    },
                    // displays the character position where the lexing error occurred, i.e. for error messages
                    showPosition: function () {
                      var pre = this.pastInput();
                      var c = new Array(pre.length + 1).join("-");
                      return pre + this.upcomingInput() + "\n" + c + "^";
                    },
                    // test the lexed token: return FALSE when not a match, otherwise return token
                    test_match: function (match, indexed_rule) {
                      var token, lines, backup;
                      if (this.options.backtrack_lexer) {
                        // save context
                        backup = {
                          yylineno: this.yylineno,
                          yylloc: {
                            first_line: this.yylloc.first_line,
                            last_line: this.last_line,
                            first_column: this.yylloc.first_column,
                            last_column: this.yylloc.last_column
                          },
                          yytext: this.yytext,
                          match: this.match,
                          matches: this.matches,
                          matched: this.matched,
                          yyleng: this.yyleng,
                          offset: this.offset,
                          _more: this._more,
                          _input: this._input,
                          yy: this.yy,
                          conditionStack: this.conditionStack.slice(0),
                          done: this.done
                        };
                        if (this.options.ranges) {
                          backup.yylloc.range = this.yylloc.range.slice(0);
                        }
                      }
                      lines = match[0].match(/(?:\r\n?|\n).*/g);
                      if (lines) {
                        this.yylineno += lines.length;
                      }
                      this.yylloc = {
                        first_line: this.yylloc.last_line,
                        last_line: this.yylineno + 1,
                        first_column: this.yylloc.last_column,
                        last_column: lines ? lines[lines.length - 1].length - lines[lines.length - 1].match(/\r?\n?/)[0].length : this.yylloc.last_column + match[0].length
                      };
                      this.yytext += match[0];
                      this.match += match[0];
                      this.matches = match;
                      this.yyleng = this.yytext.length;
                      if (this.options.ranges) {
                        this.yylloc.range = [this.offset, this.offset += this.yyleng];
                      }
                      this._more = false;
                      this._backtrack = false;
                      this._input = this._input.slice(match[0].length);
                      this.matched += match[0];
                      token = this.performAction.call(this, this.yy, this, indexed_rule, this.conditionStack[this.conditionStack.length - 1]);
                      if (this.done && this._input) {
                        this.done = false;
                      }
                      if (token) {
                        return token;
                      } else if (this._backtrack) {
                        // recover context
                        for (var k in backup) {
                          this[k] = backup[k];
                        }
                        return false; // rule action called reject() implying the next rule should be tested instead.
                      }
                      return false;
                    },
                    // return next match in input
                    next: function () {
                      if (this.done) {
                        return this.EOF;
                      }
                      if (!this._input) {
                        this.done = true;
                      }
                      var token, match, tempMatch, index;
                      if (!this._more) {
                        this.yytext = '';
                        this.match = '';
                      }
                      var rules = this._currentRules();
                      for (var i = 0; i < rules.length; i++) {
                        tempMatch = this._input.match(this.rules[rules[i]]);
                        if (tempMatch && (!match || tempMatch[0].length > match[0].length)) {
                          match = tempMatch;
                          index = i;
                          if (this.options.backtrack_lexer) {
                            token = this.test_match(tempMatch, rules[i]);
                            if (token !== false) {
                              return token;
                            } else if (this._backtrack) {
                              match = false;
                              continue; // rule action called reject() implying a rule MISmatch.
                            } else {
                              // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
                              return false;
                            }
                          } else if (!this.options.flex) {
                            break;
                          }
                        }
                      }
                      if (match) {
                        token = this.test_match(match, rules[index]);
                        if (token !== false) {
                          return token;
                        }
                        // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
                        return false;
                      }
                      if (this._input === "") {
                        return this.EOF;
                      } else {
                        return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. Unrecognized text.\n' + this.showPosition(), {
                          text: "",
                          token: null,
                          line: this.yylineno
                        });
                      }
                    },
                    // return next match that has a token
                    lex: function lex() {
                      var r = this.next();
                      if (r) {
                        return r;
                      } else {
                        return this.lex();
                      }
                    },
                    // activates a new lexer condition state (pushes the new lexer condition state onto the condition stack)
                    begin: function begin(condition) {
                      this.conditionStack.push(condition);
                    },
                    // pop the previously active lexer condition state off the condition stack
                    popState: function popState() {
                      var n = this.conditionStack.length - 1;
                      if (n > 0) {
                        return this.conditionStack.pop();
                      } else {
                        return this.conditionStack[0];
                      }
                    },
                    // produce the lexer rule set which is active for the currently active lexer condition state
                    _currentRules: function _currentRules() {
                      if (this.conditionStack.length && this.conditionStack[this.conditionStack.length - 1]) {
                        return this.conditions[this.conditionStack[this.conditionStack.length - 1]].rules;
                      } else {
                        return this.conditions["INITIAL"].rules;
                      }
                    },
                    // return the currently active lexer condition state; when an index argument is provided it produces the N-th previous condition state, if available
                    topState: function topState(n) {
                      n = this.conditionStack.length - 1 - Math.abs(n || 0);
                      if (n >= 0) {
                        return this.conditionStack[n];
                      } else {
                        return "INITIAL";
                      }
                    },
                    // alias for begin(condition)
                    pushState: function pushState(condition) {
                      this.begin(condition);
                    },
                    // return the number of states currently on the stack
                    stateStackSize: function stateStackSize() {
                      return this.conditionStack.length;
                    },
                    options: {},
                    performAction: function anonymous(yy, yy_, $avoiding_name_collisions, YY_START
                    /**/) {
                      switch ($avoiding_name_collisions) {
                        case 0:
                          return 4;
                        case 1:
                          return 14;
                        case 2:
                          return 12;
                        case 3:
                          return 15;
                        case 4:
                          return 16;
                        case 5:
                          return 22;
                        case 6:
                          return 24;
                        case 7:
                          return 28;
                        case 8:
                          return 30;
                        case 9:
                          return 18;
                        case 10:
                          yy_.yytext = yy_.yytext.substr(1, yy_.yyleng - 2);
                          return 32;
                        case 11:
                          yy_.yytext = yy_.yytext.substr(1, yy_.yyleng - 2);
                          return 33;
                        case 12:
                          return 17;
                        case 13:
                          return 31;
                      }
                    },
                    rules: [/^(?:\$)/, /^(?:\.\.)/, /^(?:\.)/, /^(?:\*)/, /^(?:[a-zA-Z_]+[a-zA-Z0-9_]*)/, /^(?:\[)/, /^(?:\])/, /^(?:,)/, /^(?:((-?(?:0|[1-9][0-9]*)))?\:((-?(?:0|[1-9][0-9]*)))?(\:((-?(?:0|[1-9][0-9]*)))?)?)/, /^(?:(-?(?:0|[1-9][0-9]*)))/, /^(?:"(?:\\["bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^"\\])*")/, /^(?:'(?:\\['bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^'\\])*')/, /^(?:\(.+?\)(?=\]))/, /^(?:\?\(.+?\)(?=\]))/],
                    conditions: {
                      "INITIAL": {
                        "rules": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13],
                        "inclusive": true
                      }
                    }
                  };
                  return lexer;
                }();
                parser.lexer = lexer;
                function Parser() {
                  this.yy = {};
                }
                Parser.prototype = parser;
                parser.Parser = Parser;
                return new Parser();
              }();
              if (typeof require !== 'undefined' && typeof exports !== 'undefined') {
                exports.parser = parser;
                exports.Parser = parser.Parser;
                exports.parse = function () {
                  return parser.parse.apply(parser, arguments);
                };
                exports.main = function commonjsMain(args) {
                  if (!args[1]) {
                    console.log('Usage: ' + args[0] + ' FILE');
                    process.exit(1);
                  }
                  var source = require('fs').readFileSync(require('path').normalize(args[1]), "utf8");
                  return exports.parser.parse(source);
                };
                if (typeof module !== 'undefined' && require.main === module) {
                  exports.main(process.argv.slice(1));
                }
              }
            }).call(this, require('_process'));
          }, {
            "_process": 14,
            "fs": 12,
            "path": 13
          }],
          2: [function (require, module, exports) {
            module.exports = {
              identifier: "[a-zA-Z_]+[a-zA-Z0-9_]*",
              integer: "-?(?:0|[1-9][0-9]*)",
              qq_string: "\"(?:\\\\[\"bfnrt/\\\\]|\\\\u[a-fA-F0-9]{4}|[^\"\\\\])*\"",
              q_string: "'(?:\\\\[\'bfnrt/\\\\]|\\\\u[a-fA-F0-9]{4}|[^\'\\\\])*'"
            };
          }, {}],
          3: [function (require, module, exports) {
            var dict = require('./dict');
            var fs = require('fs');
            var grammar = {
              lex: {
                macros: {
                  esc: "\\\\",
                  int: dict.integer
                },
                rules: [["\\$", "return 'DOLLAR'"], ["\\.\\.", "return 'DOT_DOT'"], ["\\.", "return 'DOT'"], ["\\*", "return 'STAR'"], [dict.identifier, "return 'IDENTIFIER'"], ["\\[", "return '['"], ["\\]", "return ']'"], [",", "return ','"], ["({int})?\\:({int})?(\\:({int})?)?", "return 'ARRAY_SLICE'"], ["{int}", "return 'INTEGER'"], [dict.qq_string, "yytext = yytext.substr(1,yyleng-2); return 'QQ_STRING';"], [dict.q_string, "yytext = yytext.substr(1,yyleng-2); return 'Q_STRING';"], ["\\(.+?\\)(?=\\])", "return 'SCRIPT_EXPRESSION'"], ["\\?\\(.+?\\)(?=\\])", "return 'FILTER_EXPRESSION'"]]
              },
              start: "JSON_PATH",
              bnf: {
                JSON_PATH: [['DOLLAR', 'yy.ast.set({ expression: { type: "root", value: $1 } }); yy.ast.unshift(); return yy.ast.yield()'], ['DOLLAR PATH_COMPONENTS', 'yy.ast.set({ expression: { type: "root", value: $1 } }); yy.ast.unshift(); return yy.ast.yield()'], ['LEADING_CHILD_MEMBER_EXPRESSION', 'yy.ast.unshift(); return yy.ast.yield()'], ['LEADING_CHILD_MEMBER_EXPRESSION PATH_COMPONENTS', 'yy.ast.set({ operation: "member", scope: "child", expression: { type: "identifier", value: $1 }}); yy.ast.unshift(); return yy.ast.yield()']],
                PATH_COMPONENTS: [['PATH_COMPONENT', ''], ['PATH_COMPONENTS PATH_COMPONENT', '']],
                PATH_COMPONENT: [['MEMBER_COMPONENT', 'yy.ast.set({ operation: "member" }); yy.ast.push()'], ['SUBSCRIPT_COMPONENT', 'yy.ast.set({ operation: "subscript" }); yy.ast.push() ']],
                MEMBER_COMPONENT: [['CHILD_MEMBER_COMPONENT', 'yy.ast.set({ scope: "child" })'], ['DESCENDANT_MEMBER_COMPONENT', 'yy.ast.set({ scope: "descendant" })']],
                CHILD_MEMBER_COMPONENT: [['DOT MEMBER_EXPRESSION', '']],
                LEADING_CHILD_MEMBER_EXPRESSION: [['MEMBER_EXPRESSION', 'yy.ast.set({ scope: "child", operation: "member" })']],
                DESCENDANT_MEMBER_COMPONENT: [['DOT_DOT MEMBER_EXPRESSION', '']],
                MEMBER_EXPRESSION: [['STAR', 'yy.ast.set({ expression: { type: "wildcard", value: $1 } })'], ['IDENTIFIER', 'yy.ast.set({ expression: { type: "identifier", value: $1 } })'], ['SCRIPT_EXPRESSION', 'yy.ast.set({ expression: { type: "script_expression", value: $1 } })'], ['INTEGER', 'yy.ast.set({ expression: { type: "numeric_literal", value: parseInt($1) } })'], ['END', '']],
                SUBSCRIPT_COMPONENT: [['CHILD_SUBSCRIPT_COMPONENT', 'yy.ast.set({ scope: "child" })'], ['DESCENDANT_SUBSCRIPT_COMPONENT', 'yy.ast.set({ scope: "descendant" })']],
                CHILD_SUBSCRIPT_COMPONENT: [['[ SUBSCRIPT ]', '']],
                DESCENDANT_SUBSCRIPT_COMPONENT: [['DOT_DOT [ SUBSCRIPT ]', '']],
                SUBSCRIPT: [['SUBSCRIPT_EXPRESSION', ''], ['SUBSCRIPT_EXPRESSION_LIST', '$1.length > 1? yy.ast.set({ expression: { type: "union", value: $1 } }) : $$ = $1']],
                SUBSCRIPT_EXPRESSION_LIST: [['SUBSCRIPT_EXPRESSION_LISTABLE', '$$ = [$1]'], ['SUBSCRIPT_EXPRESSION_LIST , SUBSCRIPT_EXPRESSION_LISTABLE', '$$ = $1.concat($3)']],
                SUBSCRIPT_EXPRESSION_LISTABLE: [['INTEGER', '$$ = { expression: { type: "numeric_literal", value: parseInt($1) } }; yy.ast.set($$)'], ['STRING_LITERAL', '$$ = { expression: { type: "string_literal", value: $1 } }; yy.ast.set($$)'], ['ARRAY_SLICE', '$$ = { expression: { type: "slice", value: $1 } }; yy.ast.set($$)']],
                SUBSCRIPT_EXPRESSION: [['STAR', '$$ = { expression: { type: "wildcard", value: $1 } }; yy.ast.set($$)'], ['SCRIPT_EXPRESSION', '$$ = { expression: { type: "script_expression", value: $1 } }; yy.ast.set($$)'], ['FILTER_EXPRESSION', '$$ = { expression: { type: "filter_expression", value: $1 } }; yy.ast.set($$)']],
                STRING_LITERAL: [['QQ_STRING', "$$ = $1"], ['Q_STRING', "$$ = $1"]]
              }
            };
            if (fs.readFileSync) {
              grammar.moduleInclude = fs.readFileSync(require.resolve("../include/module.js"));
              grammar.actionInclude = fs.readFileSync(require.resolve("../include/action.js"));
            }
            module.exports = grammar;
          }, {
            "./dict": 2,
            "fs": 12
          }],
          4: [function (require, module, exports) {
            var aesprim = require('./aesprim');
            var slice = require('./slice');
            var _evaluate = require('static-eval');
            var _uniq = require('underscore').uniq;
            var Handlers = function () {
              return this.initialize.apply(this, arguments);
            };
            Handlers.prototype.initialize = function () {
              this.traverse = traverser(true);
              this.descend = traverser();
            };
            Handlers.prototype.keys = Object.keys;
            Handlers.prototype.resolve = function (component) {
              var key = [component.operation, component.scope, component.expression.type].join('-');
              var method = this._fns[key];
              if (!method) throw new Error("couldn't resolve key: " + key);
              return method.bind(this);
            };
            Handlers.prototype.register = function (key, handler) {
              if (!handler instanceof Function) {
                throw new Error("handler must be a function");
              }
              this._fns[key] = handler;
            };
            Handlers.prototype._fns = {
              'member-child-identifier': function (component, partial) {
                var key = component.expression.value;
                var value = partial.value;
                if (value instanceof Object && key in value) {
                  return [{
                    value: value[key],
                    path: partial.path.concat(key)
                  }];
                }
              },
              'member-descendant-identifier': _traverse(function (key, value, ref) {
                return key == ref;
              }),
              'subscript-child-numeric_literal': _descend(function (key, value, ref) {
                return key === ref;
              }),
              'member-child-numeric_literal': _descend(function (key, value, ref) {
                return String(key) === String(ref);
              }),
              'subscript-descendant-numeric_literal': _traverse(function (key, value, ref) {
                return key === ref;
              }),
              'member-child-wildcard': _descend(function () {
                return true;
              }),
              'member-descendant-wildcard': _traverse(function () {
                return true;
              }),
              'subscript-descendant-wildcard': _traverse(function () {
                return true;
              }),
              'subscript-child-wildcard': _descend(function () {
                return true;
              }),
              'subscript-child-slice': function (component, partial) {
                if (is_array(partial.value)) {
                  var args = component.expression.value.split(':').map(_parse_nullable_int);
                  var values = partial.value.map(function (v, i) {
                    return {
                      value: v,
                      path: partial.path.concat(i)
                    };
                  });
                  return slice.apply(null, [values].concat(args));
                }
              },
              'subscript-child-union': function (component, partial) {
                var results = [];
                component.expression.value.forEach(function (component) {
                  var _component = {
                    operation: 'subscript',
                    scope: 'child',
                    expression: component.expression
                  };
                  var handler = this.resolve(_component);
                  var _results = handler(_component, partial);
                  if (_results) {
                    results = results.concat(_results);
                  }
                }, this);
                return unique(results);
              },
              'subscript-descendant-union': function (component, partial, count) {
                var jp = require('..');
                var self = this;
                var results = [];
                var nodes = jp.nodes(partial, '$..*').slice(1);
                nodes.forEach(function (node) {
                  if (results.length >= count) return;
                  component.expression.value.forEach(function (component) {
                    var _component = {
                      operation: 'subscript',
                      scope: 'child',
                      expression: component.expression
                    };
                    var handler = self.resolve(_component);
                    var _results = handler(_component, node);
                    results = results.concat(_results);
                  });
                });
                return unique(results);
              },
              'subscript-child-filter_expression': function (component, partial, count) {
                // slice out the expression from ?(expression)
                var src = component.expression.value.slice(2, -1);
                var ast = aesprim.parse(src).body[0].expression;
                var passable = function (key, value) {
                  return evaluate(ast, {
                    '@': value
                  });
                };
                return this.descend(partial, null, passable, count);
              },
              'subscript-descendant-filter_expression': function (component, partial, count) {
                // slice out the expression from ?(expression)
                var src = component.expression.value.slice(2, -1);
                var ast = aesprim.parse(src).body[0].expression;
                var passable = function (key, value) {
                  return evaluate(ast, {
                    '@': value
                  });
                };
                return this.traverse(partial, null, passable, count);
              },
              'subscript-child-script_expression': function (component, partial) {
                var exp = component.expression.value.slice(1, -1);
                return eval_recurse(partial, exp, '$[{{value}}]');
              },
              'member-child-script_expression': function (component, partial) {
                var exp = component.expression.value.slice(1, -1);
                return eval_recurse(partial, exp, '$.{{value}}');
              },
              'member-descendant-script_expression': function (component, partial) {
                var exp = component.expression.value.slice(1, -1);
                return eval_recurse(partial, exp, '$..value');
              }
            };
            Handlers.prototype._fns['subscript-child-string_literal'] = Handlers.prototype._fns['member-child-identifier'];
            Handlers.prototype._fns['member-descendant-numeric_literal'] = Handlers.prototype._fns['subscript-descendant-string_literal'] = Handlers.prototype._fns['member-descendant-identifier'];
            function eval_recurse(partial, src, template) {
              var jp = require('./index');
              var ast = aesprim.parse(src).body[0].expression;
              var value = evaluate(ast, {
                '@': partial.value
              });
              var path = template.replace(/\{\{\s*value\s*\}\}/g, value);
              var results = jp.nodes(partial.value, path);
              results.forEach(function (r) {
                r.path = partial.path.concat(r.path.slice(1));
              });
              return results;
            }
            function is_array(val) {
              return Array.isArray(val);
            }
            function is_object(val) {
              // is this a non-array, non-null object?
              return val && !(val instanceof Array) && val instanceof Object;
            }
            function traverser(recurse) {
              return function (partial, ref, passable, count) {
                var value = partial.value;
                var path = partial.path;
                var results = [];
                var descend = function (value, path) {
                  if (is_array(value)) {
                    value.forEach(function (element, index) {
                      if (results.length >= count) {
                        return;
                      }
                      if (passable(index, element, ref)) {
                        results.push({
                          path: path.concat(index),
                          value: element
                        });
                      }
                    });
                    value.forEach(function (element, index) {
                      if (results.length >= count) {
                        return;
                      }
                      if (recurse) {
                        descend(element, path.concat(index));
                      }
                    });
                  } else if (is_object(value)) {
                    this.keys(value).forEach(function (k) {
                      if (results.length >= count) {
                        return;
                      }
                      if (passable(k, value[k], ref)) {
                        results.push({
                          path: path.concat(k),
                          value: value[k]
                        });
                      }
                    });
                    this.keys(value).forEach(function (k) {
                      if (results.length >= count) {
                        return;
                      }
                      if (recurse) {
                        descend(value[k], path.concat(k));
                      }
                    });
                  }
                }.bind(this);
                descend(value, path);
                return results;
              };
            }
            function _descend(passable) {
              return function (component, partial, count) {
                return this.descend(partial, component.expression.value, passable, count);
              };
            }
            function _traverse(passable) {
              return function (component, partial, count) {
                return this.traverse(partial, component.expression.value, passable, count);
              };
            }
            function evaluate() {
              try {
                return _evaluate.apply(this, arguments);
              } catch (e) {}
            }
            function unique(results) {
              results = results.filter(function (d) {
                return d;
              });
              return _uniq(results, function (r) {
                return r.path.map(function (c) {
                  return String(c).replace('-', '--');
                }).join('-');
              });
            }
            function _parse_nullable_int(val) {
              var sval = String(val);
              return sval.match(/^-?[0-9]+$/) ? parseInt(sval) : null;
            }
            module.exports = Handlers;
          }, {
            "..": "jsonpath",
            "./aesprim": "./aesprim",
            "./index": 5,
            "./slice": 7,
            "static-eval": 15,
            "underscore": 12
          }],
          5: [function (require, module, exports) {
            var assert = require('assert');
            var dict = require('./dict');
            var Parser = require('./parser');
            var Handlers = require('./handlers');
            var JSONPath = function () {
              this.initialize.apply(this, arguments);
            };
            JSONPath.prototype.initialize = function () {
              this.parser = new Parser();
              this.handlers = new Handlers();
            };
            JSONPath.prototype.parse = function (string) {
              assert.ok(_is_string(string), "we need a path");
              return this.parser.parse(string);
            };
            JSONPath.prototype.parent = function (obj, string) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              var node = this.nodes(obj, string)[0];
              node.path.pop(); /* jshint unused:false */
              return this.value(obj, node.path);
            };
            JSONPath.prototype.apply = function (obj, string, fn) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              assert.equal(typeof fn, "function", "fn needs to be function");
              var nodes = this.nodes(obj, string).sort(function (a, b) {
                // sort nodes so we apply from the bottom up
                return b.path.length - a.path.length;
              });
              nodes.forEach(function (node) {
                var key = node.path.pop();
                var parent = this.value(obj, this.stringify(node.path));
                var val = node.value = fn.call(obj, parent[key]);
                parent[key] = val;
              }, this);
              return nodes;
            };
            JSONPath.prototype.value = function (obj, path, value) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(path, "we need a path");
              if (arguments.length >= 3) {
                var node = this.nodes(obj, path).shift();
                if (!node) return this._vivify(obj, path, value);
                var key = node.path.slice(-1).shift();
                var parent = this.parent(obj, this.stringify(node.path));
                parent[key] = value;
              }
              return this.query(obj, this.stringify(path), 1).shift();
            };
            JSONPath.prototype._vivify = function (obj, string, value) {
              var self = this;
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              var path = this.parser.parse(string).map(function (component) {
                return component.expression.value;
              });
              var setValue = function (path, value) {
                var key = path.pop();
                var node = self.value(obj, path);
                if (!node) {
                  setValue(path.concat(), typeof key === 'string' ? {} : []);
                  node = self.value(obj, path);
                }
                node[key] = value;
              };
              setValue(path, value);
              return this.query(obj, string)[0];
            };
            JSONPath.prototype.query = function (obj, string, count) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(_is_string(string), "we need a path");
              var results = this.nodes(obj, string, count).map(function (r) {
                return r.value;
              });
              return results;
            };
            JSONPath.prototype.paths = function (obj, string, count) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              var results = this.nodes(obj, string, count).map(function (r) {
                return r.path;
              });
              return results;
            };
            JSONPath.prototype.nodes = function (obj, string, count) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              if (count === 0) return [];
              var path = this.parser.parse(string);
              var handlers = this.handlers;
              var partials = [{
                path: ['$'],
                value: obj
              }];
              var matches = [];
              if (path.length && path[0].expression.type == 'root') path.shift();
              if (!path.length) return partials;
              path.forEach(function (component, index) {
                if (matches.length >= count) return;
                var handler = handlers.resolve(component);
                var _partials = [];
                partials.forEach(function (p) {
                  if (matches.length >= count) return;
                  var results = handler(component, p, count);
                  if (index == path.length - 1) {
                    // if we're through the components we're done
                    matches = matches.concat(results || []);
                  } else {
                    // otherwise accumulate and carry on through
                    _partials = _partials.concat(results || []);
                  }
                });
                partials = _partials;
              });
              return count ? matches.slice(0, count) : matches;
            };
            JSONPath.prototype.stringify = function (path) {
              assert.ok(path, "we need a path");
              var string = '$';
              var templates = {
                'descendant-member': '..{{value}}',
                'child-member': '.{{value}}',
                'descendant-subscript': '..[{{value}}]',
                'child-subscript': '[{{value}}]'
              };
              path = this._normalize(path);
              path.forEach(function (component) {
                if (component.expression.type == 'root') return;
                var key = [component.scope, component.operation].join('-');
                var template = templates[key];
                var value;
                if (component.expression.type == 'string_literal') {
                  value = JSON.stringify(component.expression.value);
                } else {
                  value = component.expression.value;
                }
                if (!template) throw new Error("couldn't find template " + key);
                string += template.replace(/{{value}}/, value);
              });
              return string;
            };
            JSONPath.prototype._normalize = function (path) {
              assert.ok(path, "we need a path");
              if (typeof path == "string") {
                return this.parser.parse(path);
              } else if (Array.isArray(path) && typeof path[0] == "string") {
                var _path = [{
                  expression: {
                    type: "root",
                    value: "$"
                  }
                }];
                path.forEach(function (component, index) {
                  if (component == '$' && index === 0) return;
                  if (typeof component == "string" && component.match("^" + dict.identifier + "$")) {
                    _path.push({
                      operation: 'member',
                      scope: 'child',
                      expression: {
                        value: component,
                        type: 'identifier'
                      }
                    });
                  } else {
                    var type = typeof component == "number" ? 'numeric_literal' : 'string_literal';
                    _path.push({
                      operation: 'subscript',
                      scope: 'child',
                      expression: {
                        value: component,
                        type: type
                      }
                    });
                  }
                });
                return _path;
              } else if (Array.isArray(path) && typeof path[0] == "object") {
                return path;
              }
              throw new Error("couldn't understand path " + path);
            };
            function _is_string(obj) {
              return Object.prototype.toString.call(obj) == '[object String]';
            }
            JSONPath.Handlers = Handlers;
            JSONPath.Parser = Parser;
            var instance = new JSONPath();
            instance.JSONPath = JSONPath;
            module.exports = instance;
          }, {
            "./dict": 2,
            "./handlers": 4,
            "./parser": 6,
            "assert": 8
          }],
          6: [function (require, module, exports) {
            var grammar = require('./grammar');
            var gparser = require('../generated/parser');
            var Parser = function () {
              var parser = new gparser.Parser();
              var _parseError = parser.parseError;
              parser.yy.parseError = function () {
                if (parser.yy.ast) {
                  parser.yy.ast.initialize();
                }
                _parseError.apply(parser, arguments);
              };
              return parser;
            };
            Parser.grammar = grammar;
            module.exports = Parser;
          }, {
            "../generated/parser": 1,
            "./grammar": 3
          }],
          7: [function (require, module, exports) {
            module.exports = function (arr, start, end, step) {
              if (typeof start == 'string') throw new Error("start cannot be a string");
              if (typeof end == 'string') throw new Error("end cannot be a string");
              if (typeof step == 'string') throw new Error("step cannot be a string");
              var len = arr.length;
              if (step === 0) throw new Error("step cannot be zero");
              step = step ? integer(step) : 1;

              // normalize negative values
              start = start < 0 ? len + start : start;
              end = end < 0 ? len + end : end;

              // default extents to extents
              start = integer(start === 0 ? 0 : !start ? step > 0 ? 0 : len - 1 : start);
              end = integer(end === 0 ? 0 : !end ? step > 0 ? len : -1 : end);

              // clamp extents
              start = step > 0 ? Math.max(0, start) : Math.min(len, start);
              end = step > 0 ? Math.min(end, len) : Math.max(-1, end);

              // return empty if extents are backwards
              if (step > 0 && end <= start) return [];
              if (step < 0 && start <= end) return [];
              var result = [];
              for (var i = start; i != end; i += step) {
                if (step < 0 && i <= end || step > 0 && i >= end) break;
                result.push(arr[i]);
              }
              return result;
            };
            function integer(val) {
              return String(val).match(/^[0-9]+$/) ? parseInt(val) : Number.isFinite(val) ? parseInt(val, 10) : 0;
            }
          }, {}],
          8: [function (require, module, exports) {
            // http://wiki.commonjs.org/wiki/Unit_Testing/1.0
            //
            // THIS IS NOT TESTED NOR LIKELY TO WORK OUTSIDE V8!
            //
            // Originally from narwhal.js (http://narwhaljs.org)
            // Copyright (c) 2009 Thomas Robinson <280north.com>
            //
            // Permission is hereby granted, free of charge, to any person obtaining a copy
            // of this software and associated documentation files (the 'Software'), to
            // deal in the Software without restriction, including without limitation the
            // rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
            // sell copies of the Software, and to permit persons to whom the Software is
            // furnished to do so, subject to the following conditions:
            //
            // The above copyright notice and this permission notice shall be included in
            // all copies or substantial portions of the Software.
            //
            // THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
            // IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
            // FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
            // AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
            // ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
            // WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

            // when used in node, this will actually load the util module we depend on
            // versus loading the builtin util module as happens otherwise
            // this is a bug in node module loading as far as I am concerned
            var util = require('util/');
            var pSlice = Array.prototype.slice;
            var hasOwn = Object.prototype.hasOwnProperty;

            // 1. The assert module provides functions that throw
            // AssertionError's when particular conditions are not met. The
            // assert module must conform to the following interface.

            var assert = module.exports = ok;

            // 2. The AssertionError is defined in assert.
            // new assert.AssertionError({ message: message,
            //                             actual: actual,
            //                             expected: expected })

            assert.AssertionError = function AssertionError(options) {
              this.name = 'AssertionError';
              this.actual = options.actual;
              this.expected = options.expected;
              this.operator = options.operator;
              if (options.message) {
                this.message = options.message;
                this.generatedMessage = false;
              } else {
                this.message = getMessage(this);
                this.generatedMessage = true;
              }
              var stackStartFunction = options.stackStartFunction || fail;
              if (Error.captureStackTrace) {
                Error.captureStackTrace(this, stackStartFunction);
              } else {
                // non v8 browsers so we can have a stacktrace
                var err = new Error();
                if (err.stack) {
                  var out = err.stack;

                  // try to strip useless frames
                  var fn_name = stackStartFunction.name;
                  var idx = out.indexOf('\n' + fn_name);
                  if (idx >= 0) {
                    // once we have located the function frame
                    // we need to strip out everything before it (and its line)
                    var next_line = out.indexOf('\n', idx + 1);
                    out = out.substring(next_line + 1);
                  }
                  this.stack = out;
                }
              }
            };

            // assert.AssertionError instanceof Error
            util.inherits(assert.AssertionError, Error);
            function replacer(key, value) {
              if (util.isUndefined(value)) {
                return '' + value;
              }
              if (util.isNumber(value) && !isFinite(value)) {
                return value.toString();
              }
              if (util.isFunction(value) || util.isRegExp(value)) {
                return value.toString();
              }
              return value;
            }
            function truncate(s, n) {
              if (util.isString(s)) {
                return s.length < n ? s : s.slice(0, n);
              } else {
                return s;
              }
            }
            function getMessage(self) {
              return truncate(JSON.stringify(self.actual, replacer), 128) + ' ' + self.operator + ' ' + truncate(JSON.stringify(self.expected, replacer), 128);
            }

            // At present only the three keys mentioned above are used and
            // understood by the spec. Implementations or sub modules can pass
            // other keys to the AssertionError's constructor - they will be
            // ignored.

            // 3. All of the following functions must throw an AssertionError
            // when a corresponding condition is not met, with a message that
            // may be undefined if not provided.  All assertion methods provide
            // both the actual and expected values to the assertion error for
            // display purposes.

            function fail(actual, expected, message, operator, stackStartFunction) {
              throw new assert.AssertionError({
                message: message,
                actual: actual,
                expected: expected,
                operator: operator,
                stackStartFunction: stackStartFunction
              });
            }

            // EXTENSION! allows for well behaved errors defined elsewhere.
            assert.fail = fail;

            // 4. Pure assertion tests whether a value is truthy, as determined
            // by !!guard.
            // assert.ok(guard, message_opt);
            // This statement is equivalent to assert.equal(true, !!guard,
            // message_opt);. To test strictly for the value true, use
            // assert.strictEqual(true, guard, message_opt);.

            function ok(value, message) {
              if (!value) fail(value, true, message, '==', assert.ok);
            }
            assert.ok = ok;

            // 5. The equality assertion tests shallow, coercive equality with
            // ==.
            // assert.equal(actual, expected, message_opt);

            assert.equal = function equal(actual, expected, message) {
              if (actual != expected) fail(actual, expected, message, '==', assert.equal);
            };

            // 6. The non-equality assertion tests for whether two objects are not equal
            // with != assert.notEqual(actual, expected, message_opt);

            assert.notEqual = function notEqual(actual, expected, message) {
              if (actual == expected) {
                fail(actual, expected, message, '!=', assert.notEqual);
              }
            };

            // 7. The equivalence assertion tests a deep equality relation.
            // assert.deepEqual(actual, expected, message_opt);

            assert.deepEqual = function deepEqual(actual, expected, message) {
              if (!_deepEqual(actual, expected)) {
                fail(actual, expected, message, 'deepEqual', assert.deepEqual);
              }
            };
            function _deepEqual(actual, expected) {
              // 7.1. All identical values are equivalent, as determined by ===.
              if (actual === expected) {
                return true;
              } else if (util.isBuffer(actual) && util.isBuffer(expected)) {
                if (actual.length != expected.length) return false;
                for (var i = 0; i < actual.length; i++) {
                  if (actual[i] !== expected[i]) return false;
                }
                return true;

                // 7.2. If the expected value is a Date object, the actual value is
                // equivalent if it is also a Date object that refers to the same time.
              } else if (util.isDate(actual) && util.isDate(expected)) {
                return actual.getTime() === expected.getTime();

                // 7.3 If the expected value is a RegExp object, the actual value is
                // equivalent if it is also a RegExp object with the same source and
                // properties (`global`, `multiline`, `lastIndex`, `ignoreCase`).
              } else if (util.isRegExp(actual) && util.isRegExp(expected)) {
                return actual.source === expected.source && actual.global === expected.global && actual.multiline === expected.multiline && actual.lastIndex === expected.lastIndex && actual.ignoreCase === expected.ignoreCase;

                // 7.4. Other pairs that do not both pass typeof value == 'object',
                // equivalence is determined by ==.
              } else if (!util.isObject(actual) && !util.isObject(expected)) {
                return actual == expected;

                // 7.5 For all other Object pairs, including Array objects, equivalence is
                // determined by having the same number of owned properties (as verified
                // with Object.prototype.hasOwnProperty.call), the same set of keys
                // (although not necessarily the same order), equivalent values for every
                // corresponding key, and an identical 'prototype' property. Note: this
                // accounts for both named and indexed properties on Arrays.
              } else {
                return objEquiv(actual, expected);
              }
            }
            function isArguments(object) {
              return Object.prototype.toString.call(object) == '[object Arguments]';
            }
            function objEquiv(a, b) {
              if (util.isNullOrUndefined(a) || util.isNullOrUndefined(b)) return false;
              // an identical 'prototype' property.
              if (a.prototype !== b.prototype) return false;
              // if one is a primitive, the other must be same
              if (util.isPrimitive(a) || util.isPrimitive(b)) {
                return a === b;
              }
              var aIsArgs = isArguments(a),
                bIsArgs = isArguments(b);
              if (aIsArgs && !bIsArgs || !aIsArgs && bIsArgs) return false;
              if (aIsArgs) {
                a = pSlice.call(a);
                b = pSlice.call(b);
                return _deepEqual(a, b);
              }
              var ka = objectKeys(a),
                kb = objectKeys(b),
                key,
                i;
              // having the same number of owned properties (keys incorporates
              // hasOwnProperty)
              if (ka.length != kb.length) return false;
              //the same set of keys (although not necessarily the same order),
              ka.sort();
              kb.sort();
              //~~~cheap key test
              for (i = ka.length - 1; i >= 0; i--) {
                if (ka[i] != kb[i]) return false;
              }
              //equivalent values for every corresponding key, and
              //~~~possibly expensive deep test
              for (i = ka.length - 1; i >= 0; i--) {
                key = ka[i];
                if (!_deepEqual(a[key], b[key])) return false;
              }
              return true;
            }

            // 8. The non-equivalence assertion tests for any deep inequality.
            // assert.notDeepEqual(actual, expected, message_opt);

            assert.notDeepEqual = function notDeepEqual(actual, expected, message) {
              if (_deepEqual(actual, expected)) {
                fail(actual, expected, message, 'notDeepEqual', assert.notDeepEqual);
              }
            };

            // 9. The strict equality assertion tests strict equality, as determined by ===.
            // assert.strictEqual(actual, expected, message_opt);

            assert.strictEqual = function strictEqual(actual, expected, message) {
              if (actual !== expected) {
                fail(actual, expected, message, '===', assert.strictEqual);
              }
            };

            // 10. The strict non-equality assertion tests for strict inequality, as
            // determined by !==.  assert.notStrictEqual(actual, expected, message_opt);

            assert.notStrictEqual = function notStrictEqual(actual, expected, message) {
              if (actual === expected) {
                fail(actual, expected, message, '!==', assert.notStrictEqual);
              }
            };
            function expectedException(actual, expected) {
              if (!actual || !expected) {
                return false;
              }
              if (Object.prototype.toString.call(expected) == '[object RegExp]') {
                return expected.test(actual);
              } else if (actual instanceof expected) {
                return true;
              } else if (expected.call({}, actual) === true) {
                return true;
              }
              return false;
            }
            function _throws(shouldThrow, block, expected, message) {
              var actual;
              if (util.isString(expected)) {
                message = expected;
                expected = null;
              }
              try {
                block();
              } catch (e) {
                actual = e;
              }
              message = (expected && expected.name ? ' (' + expected.name + ').' : '.') + (message ? ' ' + message : '.');
              if (shouldThrow && !actual) {
                fail(actual, expected, 'Missing expected exception' + message);
              }
              if (!shouldThrow && expectedException(actual, expected)) {
                fail(actual, expected, 'Got unwanted exception' + message);
              }
              if (shouldThrow && actual && expected && !expectedException(actual, expected) || !shouldThrow && actual) {
                throw actual;
              }
            }

            // 11. Expected to throw an error:
            // assert.throws(block, Error_opt, message_opt);

            assert.throws = function (block, /*optional*/error, /*optional*/message) {
              _throws.apply(this, [true].concat(pSlice.call(arguments)));
            };

            // EXTENSION! This is annoying to write outside this module.
            assert.doesNotThrow = function (block, /*optional*/message) {
              _throws.apply(this, [false].concat(pSlice.call(arguments)));
            };
            assert.ifError = function (err) {
              if (err) {
                throw err;
              }
            };
            var objectKeys = Object.keys || function (obj) {
              var keys = [];
              for (var key in obj) {
                if (hasOwn.call(obj, key)) keys.push(key);
              }
              return keys;
            };
          }, {
            "util/": 11
          }],
          9: [function (require, module, exports) {
            if (typeof Object.create === 'function') {
              // implementation from standard node.js 'util' module
              module.exports = function inherits(ctor, superCtor) {
                ctor.super_ = superCtor;
                ctor.prototype = Object.create(superCtor.prototype, {
                  constructor: {
                    value: ctor,
                    enumerable: false,
                    writable: true,
                    configurable: true
                  }
                });
              };
            } else {
              // old school shim for old browsers
              module.exports = function inherits(ctor, superCtor) {
                ctor.super_ = superCtor;
                var TempCtor = function () {};
                TempCtor.prototype = superCtor.prototype;
                ctor.prototype = new TempCtor();
                ctor.prototype.constructor = ctor;
              };
            }
          }, {}],
          10: [function (require, module, exports) {
            module.exports = function isBuffer(arg) {
              return arg && typeof arg === 'object' && typeof arg.copy === 'function' && typeof arg.fill === 'function' && typeof arg.readUInt8 === 'function';
            };
          }, {}],
          11: [function (require, module, exports) {
            (function (process, global) {
              // Copyright Joyent, Inc. and other Node contributors.
              //
              // Permission is hereby granted, free of charge, to any person obtaining a
              // copy of this software and associated documentation files (the
              // "Software"), to deal in the Software without restriction, including
              // without limitation the rights to use, copy, modify, merge, publish,
              // distribute, sublicense, and/or sell copies of the Software, and to permit
              // persons to whom the Software is furnished to do so, subject to the
              // following conditions:
              //
              // The above copyright notice and this permission notice shall be included
              // in all copies or substantial portions of the Software.
              //
              // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
              // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
              // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
              // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
              // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
              // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
              // USE OR OTHER DEALINGS IN THE SOFTWARE.

              var formatRegExp = /%[sdj%]/g;
              exports.format = function (f) {
                if (!isString(f)) {
                  var objects = [];
                  for (var i = 0; i < arguments.length; i++) {
                    objects.push(inspect(arguments[i]));
                  }
                  return objects.join(' ');
                }
                var i = 1;
                var args = arguments;
                var len = args.length;
                var str = String(f).replace(formatRegExp, function (x) {
                  if (x === '%%') return '%';
                  if (i >= len) return x;
                  switch (x) {
                    case '%s':
                      return String(args[i++]);
                    case '%d':
                      return Number(args[i++]);
                    case '%j':
                      try {
                        return JSON.stringify(args[i++]);
                      } catch (_) {
                        return '[Circular]';
                      }
                    default:
                      return x;
                  }
                });
                for (var x = args[i]; i < len; x = args[++i]) {
                  if (isNull(x) || !isObject(x)) {
                    str += ' ' + x;
                  } else {
                    str += ' ' + inspect(x);
                  }
                }
                return str;
              };

              // Mark that a method should not be used.
              // Returns a modified function which warns once by default.
              // If --no-deprecation is set, then it is a no-op.
              exports.deprecate = function (fn, msg) {
                // Allow for deprecating things in the process of starting up.
                if (isUndefined(global.process)) {
                  return function () {
                    return exports.deprecate(fn, msg).apply(this, arguments);
                  };
                }
                if (process.noDeprecation === true) {
                  return fn;
                }
                var warned = false;
                function deprecated() {
                  if (!warned) {
                    if (process.throwDeprecation) {
                      throw new Error(msg);
                    } else if (process.traceDeprecation) {
                      console.trace(msg);
                    } else {
                      console.error(msg);
                    }
                    warned = true;
                  }
                  return fn.apply(this, arguments);
                }
                return deprecated;
              };
              var debugs = {};
              var debugEnviron;
              exports.debuglog = function (set) {
                if (isUndefined(debugEnviron)) debugEnviron = process.env.NODE_DEBUG || '';
                set = set.toUpperCase();
                if (!debugs[set]) {
                  if (new RegExp('\\b' + set + '\\b', 'i').test(debugEnviron)) {
                    var pid = process.pid;
                    debugs[set] = function () {
                      var msg = exports.format.apply(exports, arguments);
                      console.error('%s %d: %s', set, pid, msg);
                    };
                  } else {
                    debugs[set] = function () {};
                  }
                }
                return debugs[set];
              };

              /**
               * Echos the value of a value. Trys to print the value out
               * in the best way possible given the different types.
               *
               * @param {Object} obj The object to print out.
               * @param {Object} opts Optional options object that alters the output.
               */
              /* legacy: obj, showHidden, depth, colors*/
              function inspect(obj, opts) {
                // default options
                var ctx = {
                  seen: [],
                  stylize: stylizeNoColor
                };
                // legacy...
                if (arguments.length >= 3) ctx.depth = arguments[2];
                if (arguments.length >= 4) ctx.colors = arguments[3];
                if (isBoolean(opts)) {
                  // legacy...
                  ctx.showHidden = opts;
                } else if (opts) {
                  // got an "options" object
                  exports._extend(ctx, opts);
                }
                // set default options
                if (isUndefined(ctx.showHidden)) ctx.showHidden = false;
                if (isUndefined(ctx.depth)) ctx.depth = 2;
                if (isUndefined(ctx.colors)) ctx.colors = false;
                if (isUndefined(ctx.customInspect)) ctx.customInspect = true;
                if (ctx.colors) ctx.stylize = stylizeWithColor;
                return formatValue(ctx, obj, ctx.depth);
              }
              exports.inspect = inspect;

              // http://en.wikipedia.org/wiki/ANSI_escape_code#graphics
              inspect.colors = {
                'bold': [1, 22],
                'italic': [3, 23],
                'underline': [4, 24],
                'inverse': [7, 27],
                'white': [37, 39],
                'grey': [90, 39],
                'black': [30, 39],
                'blue': [34, 39],
                'cyan': [36, 39],
                'green': [32, 39],
                'magenta': [35, 39],
                'red': [31, 39],
                'yellow': [33, 39]
              };

              // Don't use 'blue' not visible on cmd.exe
              inspect.styles = {
                'special': 'cyan',
                'number': 'yellow',
                'boolean': 'yellow',
                'undefined': 'grey',
                'null': 'bold',
                'string': 'green',
                'date': 'magenta',
                // "name": intentionally not styling
                'regexp': 'red'
              };
              function stylizeWithColor(str, styleType) {
                var style = inspect.styles[styleType];
                if (style) {
                  return '\u001b[' + inspect.colors[style][0] + 'm' + str + '\u001b[' + inspect.colors[style][1] + 'm';
                } else {
                  return str;
                }
              }
              function stylizeNoColor(str, styleType) {
                return str;
              }
              function arrayToHash(array) {
                var hash = {};
                array.forEach(function (val, idx) {
                  hash[val] = true;
                });
                return hash;
              }
              function formatValue(ctx, value, recurseTimes) {
                // Provide a hook for user-specified inspect functions.
                // Check that value is an object with an inspect function on it
                if (ctx.customInspect && value && isFunction(value.inspect) &&
                // Filter out the util module, it's inspect function is special
                value.inspect !== exports.inspect &&
                // Also filter out any prototype objects using the circular check.
                !(value.constructor && value.constructor.prototype === value)) {
                  var ret = value.inspect(recurseTimes, ctx);
                  if (!isString(ret)) {
                    ret = formatValue(ctx, ret, recurseTimes);
                  }
                  return ret;
                }

                // Primitive types cannot have properties
                var primitive = formatPrimitive(ctx, value);
                if (primitive) {
                  return primitive;
                }

                // Look up the keys of the object.
                var keys = Object.keys(value);
                var visibleKeys = arrayToHash(keys);
                if (ctx.showHidden) {
                  keys = Object.getOwnPropertyNames(value);
                }

                // IE doesn't make error fields non-enumerable
                // http://msdn.microsoft.com/en-us/library/ie/dww52sbt(v=vs.94).aspx
                if (isError(value) && (keys.indexOf('message') >= 0 || keys.indexOf('description') >= 0)) {
                  return formatError(value);
                }

                // Some type of object without properties can be shortcutted.
                if (keys.length === 0) {
                  if (isFunction(value)) {
                    var name = value.name ? ': ' + value.name : '';
                    return ctx.stylize('[Function' + name + ']', 'special');
                  }
                  if (isRegExp(value)) {
                    return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
                  }
                  if (isDate(value)) {
                    return ctx.stylize(Date.prototype.toString.call(value), 'date');
                  }
                  if (isError(value)) {
                    return formatError(value);
                  }
                }
                var base = '',
                  array = false,
                  braces = ['{', '}'];

                // Make Array say that they are Array
                if (isArray(value)) {
                  array = true;
                  braces = ['[', ']'];
                }

                // Make functions say that they are functions
                if (isFunction(value)) {
                  var n = value.name ? ': ' + value.name : '';
                  base = ' [Function' + n + ']';
                }

                // Make RegExps say that they are RegExps
                if (isRegExp(value)) {
                  base = ' ' + RegExp.prototype.toString.call(value);
                }

                // Make dates with properties first say the date
                if (isDate(value)) {
                  base = ' ' + Date.prototype.toUTCString.call(value);
                }

                // Make error with message first say the error
                if (isError(value)) {
                  base = ' ' + formatError(value);
                }
                if (keys.length === 0 && (!array || value.length == 0)) {
                  return braces[0] + base + braces[1];
                }
                if (recurseTimes < 0) {
                  if (isRegExp(value)) {
                    return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
                  } else {
                    return ctx.stylize('[Object]', 'special');
                  }
                }
                ctx.seen.push(value);
                var output;
                if (array) {
                  output = formatArray(ctx, value, recurseTimes, visibleKeys, keys);
                } else {
                  output = keys.map(function (key) {
                    return formatProperty(ctx, value, recurseTimes, visibleKeys, key, array);
                  });
                }
                ctx.seen.pop();
                return reduceToSingleString(output, base, braces);
              }
              function formatPrimitive(ctx, value) {
                if (isUndefined(value)) return ctx.stylize('undefined', 'undefined');
                if (isString(value)) {
                  var simple = '\'' + JSON.stringify(value).replace(/^"|"$/g, '').replace(/'/g, "\\'").replace(/\\"/g, '"') + '\'';
                  return ctx.stylize(simple, 'string');
                }
                if (isNumber(value)) return ctx.stylize('' + value, 'number');
                if (isBoolean(value)) return ctx.stylize('' + value, 'boolean');
                // For some reason typeof null is "object", so special case here.
                if (isNull(value)) return ctx.stylize('null', 'null');
              }
              function formatError(value) {
                return '[' + Error.prototype.toString.call(value) + ']';
              }
              function formatArray(ctx, value, recurseTimes, visibleKeys, keys) {
                var output = [];
                for (var i = 0, l = value.length; i < l; ++i) {
                  if (hasOwnProperty(value, String(i))) {
                    output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, String(i), true));
                  } else {
                    output.push('');
                  }
                }
                keys.forEach(function (key) {
                  if (!key.match(/^\d+$/)) {
                    output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, key, true));
                  }
                });
                return output;
              }
              function formatProperty(ctx, value, recurseTimes, visibleKeys, key, array) {
                var name, str, desc;
                desc = Object.getOwnPropertyDescriptor(value, key) || {
                  value: value[key]
                };
                if (desc.get) {
                  if (desc.set) {
                    str = ctx.stylize('[Getter/Setter]', 'special');
                  } else {
                    str = ctx.stylize('[Getter]', 'special');
                  }
                } else {
                  if (desc.set) {
                    str = ctx.stylize('[Setter]', 'special');
                  }
                }
                if (!hasOwnProperty(visibleKeys, key)) {
                  name = '[' + key + ']';
                }
                if (!str) {
                  if (ctx.seen.indexOf(desc.value) < 0) {
                    if (isNull(recurseTimes)) {
                      str = formatValue(ctx, desc.value, null);
                    } else {
                      str = formatValue(ctx, desc.value, recurseTimes - 1);
                    }
                    if (str.indexOf('\n') > -1) {
                      if (array) {
                        str = str.split('\n').map(function (line) {
                          return '  ' + line;
                        }).join('\n').substr(2);
                      } else {
                        str = '\n' + str.split('\n').map(function (line) {
                          return '   ' + line;
                        }).join('\n');
                      }
                    }
                  } else {
                    str = ctx.stylize('[Circular]', 'special');
                  }
                }
                if (isUndefined(name)) {
                  if (array && key.match(/^\d+$/)) {
                    return str;
                  }
                  name = JSON.stringify('' + key);
                  if (name.match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)) {
                    name = name.substr(1, name.length - 2);
                    name = ctx.stylize(name, 'name');
                  } else {
                    name = name.replace(/'/g, "\\'").replace(/\\"/g, '"').replace(/(^"|"$)/g, "'");
                    name = ctx.stylize(name, 'string');
                  }
                }
                return name + ': ' + str;
              }
              function reduceToSingleString(output, base, braces) {
                var length = output.reduce(function (prev, cur) {
                  if (cur.indexOf('\n') >= 0) ;
                  return prev + cur.replace(/\u001b\[\d\d?m/g, '').length + 1;
                }, 0);
                if (length > 60) {
                  return braces[0] + (base === '' ? '' : base + '\n ') + ' ' + output.join(',\n  ') + ' ' + braces[1];
                }
                return braces[0] + base + ' ' + output.join(', ') + ' ' + braces[1];
              }

              // NOTE: These type checking functions intentionally don't use `instanceof`
              // because it is fragile and can be easily faked with `Object.create()`.
              function isArray(ar) {
                return Array.isArray(ar);
              }
              exports.isArray = isArray;
              function isBoolean(arg) {
                return typeof arg === 'boolean';
              }
              exports.isBoolean = isBoolean;
              function isNull(arg) {
                return arg === null;
              }
              exports.isNull = isNull;
              function isNullOrUndefined(arg) {
                return arg == null;
              }
              exports.isNullOrUndefined = isNullOrUndefined;
              function isNumber(arg) {
                return typeof arg === 'number';
              }
              exports.isNumber = isNumber;
              function isString(arg) {
                return typeof arg === 'string';
              }
              exports.isString = isString;
              function isSymbol(arg) {
                return typeof arg === 'symbol';
              }
              exports.isSymbol = isSymbol;
              function isUndefined(arg) {
                return arg === void 0;
              }
              exports.isUndefined = isUndefined;
              function isRegExp(re) {
                return isObject(re) && objectToString(re) === '[object RegExp]';
              }
              exports.isRegExp = isRegExp;
              function isObject(arg) {
                return typeof arg === 'object' && arg !== null;
              }
              exports.isObject = isObject;
              function isDate(d) {
                return isObject(d) && objectToString(d) === '[object Date]';
              }
              exports.isDate = isDate;
              function isError(e) {
                return isObject(e) && (objectToString(e) === '[object Error]' || e instanceof Error);
              }
              exports.isError = isError;
              function isFunction(arg) {
                return typeof arg === 'function';
              }
              exports.isFunction = isFunction;
              function isPrimitive(arg) {
                return arg === null || typeof arg === 'boolean' || typeof arg === 'number' || typeof arg === 'string' || typeof arg === 'symbol' ||
                // ES6 symbol
                typeof arg === 'undefined';
              }
              exports.isPrimitive = isPrimitive;
              exports.isBuffer = require('./support/isBuffer');
              function objectToString(o) {
                return Object.prototype.toString.call(o);
              }
              function pad(n) {
                return n < 10 ? '0' + n.toString(10) : n.toString(10);
              }
              var months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

              // 26 Feb 16:19:34
              function timestamp() {
                var d = new Date();
                var time = [pad(d.getHours()), pad(d.getMinutes()), pad(d.getSeconds())].join(':');
                return [d.getDate(), months[d.getMonth()], time].join(' ');
              }

              // log is just a thin wrapper to console.log that prepends a timestamp
              exports.log = function () {
                console.log('%s - %s', timestamp(), exports.format.apply(exports, arguments));
              };

              /**
               * Inherit the prototype methods from one constructor into another.
               *
               * The Function.prototype.inherits from lang.js rewritten as a standalone
               * function (not on Function.prototype). NOTE: If this file is to be loaded
               * during bootstrapping this function needs to be rewritten using some native
               * functions as prototype setup using normal JavaScript does not work as
               * expected during bootstrapping (see mirror.js in r114903).
               *
               * @param {function} ctor Constructor function which needs to inherit the
               *     prototype.
               * @param {function} superCtor Constructor function to inherit prototype from.
               */
              exports.inherits = require('inherits');
              exports._extend = function (origin, add) {
                // Don't do anything if add isn't an object
                if (!add || !isObject(add)) return origin;
                var keys = Object.keys(add);
                var i = keys.length;
                while (i--) {
                  origin[keys[i]] = add[keys[i]];
                }
                return origin;
              };
              function hasOwnProperty(obj, prop) {
                return Object.prototype.hasOwnProperty.call(obj, prop);
              }
            }).call(this, require('_process'), typeof commonjsGlobal !== "undefined" ? commonjsGlobal : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {});
          }, {
            "./support/isBuffer": 10,
            "_process": 14,
            "inherits": 9
          }],
          12: [function (require, module, exports) {}, {}],
          13: [function (require, module, exports) {
            (function (process) {
              // .dirname, .basename, and .extname methods are extracted from Node.js v8.11.1,
              // backported and transplited with Babel, with backwards-compat fixes

              // Copyright Joyent, Inc. and other Node contributors.
              //
              // Permission is hereby granted, free of charge, to any person obtaining a
              // copy of this software and associated documentation files (the
              // "Software"), to deal in the Software without restriction, including
              // without limitation the rights to use, copy, modify, merge, publish,
              // distribute, sublicense, and/or sell copies of the Software, and to permit
              // persons to whom the Software is furnished to do so, subject to the
              // following conditions:
              //
              // The above copyright notice and this permission notice shall be included
              // in all copies or substantial portions of the Software.
              //
              // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
              // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
              // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
              // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
              // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
              // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
              // USE OR OTHER DEALINGS IN THE SOFTWARE.

              // resolves . and .. elements in a path array with directory names there
              // must be no slashes, empty elements, or device names (c:\) in the array
              // (so also no leading and trailing slashes - it does not distinguish
              // relative and absolute paths)
              function normalizeArray(parts, allowAboveRoot) {
                // if the path tries to go above the root, `up` ends up > 0
                var up = 0;
                for (var i = parts.length - 1; i >= 0; i--) {
                  var last = parts[i];
                  if (last === '.') {
                    parts.splice(i, 1);
                  } else if (last === '..') {
                    parts.splice(i, 1);
                    up++;
                  } else if (up) {
                    parts.splice(i, 1);
                    up--;
                  }
                }

                // if the path is allowed to go above the root, restore leading ..s
                if (allowAboveRoot) {
                  for (; up--; up) {
                    parts.unshift('..');
                  }
                }
                return parts;
              }

              // path.resolve([from ...], to)
              // posix version
              exports.resolve = function () {
                var resolvedPath = '',
                  resolvedAbsolute = false;
                for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
                  var path = i >= 0 ? arguments[i] : process.cwd();

                  // Skip empty and invalid entries
                  if (typeof path !== 'string') {
                    throw new TypeError('Arguments to path.resolve must be strings');
                  } else if (!path) {
                    continue;
                  }
                  resolvedPath = path + '/' + resolvedPath;
                  resolvedAbsolute = path.charAt(0) === '/';
                }

                // At this point the path should be resolved to a full absolute path, but
                // handle relative paths to be safe (might happen when process.cwd() fails)

                // Normalize the path
                resolvedPath = normalizeArray(filter(resolvedPath.split('/'), function (p) {
                  return !!p;
                }), !resolvedAbsolute).join('/');
                return (resolvedAbsolute ? '/' : '') + resolvedPath || '.';
              };

              // path.normalize(path)
              // posix version
              exports.normalize = function (path) {
                var isAbsolute = exports.isAbsolute(path),
                  trailingSlash = substr(path, -1) === '/';

                // Normalize the path
                path = normalizeArray(filter(path.split('/'), function (p) {
                  return !!p;
                }), !isAbsolute).join('/');
                if (!path && !isAbsolute) {
                  path = '.';
                }
                if (path && trailingSlash) {
                  path += '/';
                }
                return (isAbsolute ? '/' : '') + path;
              };

              // posix version
              exports.isAbsolute = function (path) {
                return path.charAt(0) === '/';
              };

              // posix version
              exports.join = function () {
                var paths = Array.prototype.slice.call(arguments, 0);
                return exports.normalize(filter(paths, function (p, index) {
                  if (typeof p !== 'string') {
                    throw new TypeError('Arguments to path.join must be strings');
                  }
                  return p;
                }).join('/'));
              };

              // path.relative(from, to)
              // posix version
              exports.relative = function (from, to) {
                from = exports.resolve(from).substr(1);
                to = exports.resolve(to).substr(1);
                function trim(arr) {
                  var start = 0;
                  for (; start < arr.length; start++) {
                    if (arr[start] !== '') break;
                  }
                  var end = arr.length - 1;
                  for (; end >= 0; end--) {
                    if (arr[end] !== '') break;
                  }
                  if (start > end) return [];
                  return arr.slice(start, end - start + 1);
                }
                var fromParts = trim(from.split('/'));
                var toParts = trim(to.split('/'));
                var length = Math.min(fromParts.length, toParts.length);
                var samePartsLength = length;
                for (var i = 0; i < length; i++) {
                  if (fromParts[i] !== toParts[i]) {
                    samePartsLength = i;
                    break;
                  }
                }
                var outputParts = [];
                for (var i = samePartsLength; i < fromParts.length; i++) {
                  outputParts.push('..');
                }
                outputParts = outputParts.concat(toParts.slice(samePartsLength));
                return outputParts.join('/');
              };
              exports.sep = '/';
              exports.delimiter = ':';
              exports.dirname = function (path) {
                if (typeof path !== 'string') path = path + '';
                if (path.length === 0) return '.';
                var code = path.charCodeAt(0);
                var hasRoot = code === 47 /*/*/;
                var end = -1;
                var matchedSlash = true;
                for (var i = path.length - 1; i >= 1; --i) {
                  code = path.charCodeAt(i);
                  if (code === 47 /*/*/) {
                    if (!matchedSlash) {
                      end = i;
                      break;
                    }
                  } else {
                    // We saw the first non-path separator
                    matchedSlash = false;
                  }
                }
                if (end === -1) return hasRoot ? '/' : '.';
                if (hasRoot && end === 1) {
                  // return '//';
                  // Backwards-compat fix:
                  return '/';
                }
                return path.slice(0, end);
              };
              function basename(path) {
                if (typeof path !== 'string') path = path + '';
                var start = 0;
                var end = -1;
                var matchedSlash = true;
                var i;
                for (i = path.length - 1; i >= 0; --i) {
                  if (path.charCodeAt(i) === 47 /*/*/) {
                    // If we reached a path separator that was not part of a set of path
                    // separators at the end of the string, stop now
                    if (!matchedSlash) {
                      start = i + 1;
                      break;
                    }
                  } else if (end === -1) {
                    // We saw the first non-path separator, mark this as the end of our
                    // path component
                    matchedSlash = false;
                    end = i + 1;
                  }
                }
                if (end === -1) return '';
                return path.slice(start, end);
              }

              // Uses a mixed approach for backwards-compatibility, as ext behavior changed
              // in new Node.js versions, so only basename() above is backported here
              exports.basename = function (path, ext) {
                var f = basename(path);
                if (ext && f.substr(-1 * ext.length) === ext) {
                  f = f.substr(0, f.length - ext.length);
                }
                return f;
              };
              exports.extname = function (path) {
                if (typeof path !== 'string') path = path + '';
                var startDot = -1;
                var startPart = 0;
                var end = -1;
                var matchedSlash = true;
                // Track the state of characters (if any) we see before our first dot and
                // after any path separator we find
                var preDotState = 0;
                for (var i = path.length - 1; i >= 0; --i) {
                  var code = path.charCodeAt(i);
                  if (code === 47 /*/*/) {
                    // If we reached a path separator that was not part of a set of path
                    // separators at the end of the string, stop now
                    if (!matchedSlash) {
                      startPart = i + 1;
                      break;
                    }
                    continue;
                  }
                  if (end === -1) {
                    // We saw the first non-path separator, mark this as the end of our
                    // extension
                    matchedSlash = false;
                    end = i + 1;
                  }
                  if (code === 46 /*.*/) {
                    // If this is our first dot, mark it as the start of our extension
                    if (startDot === -1) startDot = i;else if (preDotState !== 1) preDotState = 1;
                  } else if (startDot !== -1) {
                    // We saw a non-dot and non-path separator before our dot, so we should
                    // have a good chance at having a non-empty extension
                    preDotState = -1;
                  }
                }
                if (startDot === -1 || end === -1 ||
                // We saw a non-dot character immediately before the dot
                preDotState === 0 ||
                // The (right-most) trimmed path component is exactly '..'
                preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
                  return '';
                }
                return path.slice(startDot, end);
              };
              function filter(xs, f) {
                if (xs.filter) return xs.filter(f);
                var res = [];
                for (var i = 0; i < xs.length; i++) {
                  if (f(xs[i], i, xs)) res.push(xs[i]);
                }
                return res;
              }

              // String.prototype.substr - negative index don't work in IE8
              var substr = 'ab'.substr(-1) === 'b' ? function (str, start, len) {
                return str.substr(start, len);
              } : function (str, start, len) {
                if (start < 0) start = str.length + start;
                return str.substr(start, len);
              };
            }).call(this, require('_process'));
          }, {
            "_process": 14
          }],
          14: [function (require, module, exports) {
            // shim for using process in browser
            var process = module.exports = {};

            // cached from whatever global is present so that test runners that stub it
            // don't break things.  But we need to wrap it in a try catch in case it is
            // wrapped in strict mode code which doesn't define any globals.  It's inside a
            // function because try/catches deoptimize in certain engines.

            var cachedSetTimeout;
            var cachedClearTimeout;
            function defaultSetTimout() {
              throw new Error('setTimeout has not been defined');
            }
            function defaultClearTimeout() {
              throw new Error('clearTimeout has not been defined');
            }
            (function () {
              try {
                if (typeof setTimeout === 'function') {
                  cachedSetTimeout = setTimeout;
                } else {
                  cachedSetTimeout = defaultSetTimout;
                }
              } catch (e) {
                cachedSetTimeout = defaultSetTimout;
              }
              try {
                if (typeof clearTimeout === 'function') {
                  cachedClearTimeout = clearTimeout;
                } else {
                  cachedClearTimeout = defaultClearTimeout;
                }
              } catch (e) {
                cachedClearTimeout = defaultClearTimeout;
              }
            })();
            function runTimeout(fun) {
              if (cachedSetTimeout === setTimeout) {
                //normal enviroments in sane situations
                return setTimeout(fun, 0);
              }
              // if setTimeout wasn't available but was latter defined
              if ((cachedSetTimeout === defaultSetTimout || !cachedSetTimeout) && setTimeout) {
                cachedSetTimeout = setTimeout;
                return setTimeout(fun, 0);
              }
              try {
                // when when somebody has screwed with setTimeout but no I.E. maddness
                return cachedSetTimeout(fun, 0);
              } catch (e) {
                try {
                  // When we are in I.E. but the script has been evaled so I.E. doesn't trust the global object when called normally
                  return cachedSetTimeout.call(null, fun, 0);
                } catch (e) {
                  // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error
                  return cachedSetTimeout.call(this, fun, 0);
                }
              }
            }
            function runClearTimeout(marker) {
              if (cachedClearTimeout === clearTimeout) {
                //normal enviroments in sane situations
                return clearTimeout(marker);
              }
              // if clearTimeout wasn't available but was latter defined
              if ((cachedClearTimeout === defaultClearTimeout || !cachedClearTimeout) && clearTimeout) {
                cachedClearTimeout = clearTimeout;
                return clearTimeout(marker);
              }
              try {
                // when when somebody has screwed with setTimeout but no I.E. maddness
                return cachedClearTimeout(marker);
              } catch (e) {
                try {
                  // When we are in I.E. but the script has been evaled so I.E. doesn't  trust the global object when called normally
                  return cachedClearTimeout.call(null, marker);
                } catch (e) {
                  // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error.
                  // Some versions of I.E. have different rules for clearTimeout vs setTimeout
                  return cachedClearTimeout.call(this, marker);
                }
              }
            }
            var queue = [];
            var draining = false;
            var currentQueue;
            var queueIndex = -1;
            function cleanUpNextTick() {
              if (!draining || !currentQueue) {
                return;
              }
              draining = false;
              if (currentQueue.length) {
                queue = currentQueue.concat(queue);
              } else {
                queueIndex = -1;
              }
              if (queue.length) {
                drainQueue();
              }
            }
            function drainQueue() {
              if (draining) {
                return;
              }
              var timeout = runTimeout(cleanUpNextTick);
              draining = true;
              var len = queue.length;
              while (len) {
                currentQueue = queue;
                queue = [];
                while (++queueIndex < len) {
                  if (currentQueue) {
                    currentQueue[queueIndex].run();
                  }
                }
                queueIndex = -1;
                len = queue.length;
              }
              currentQueue = null;
              draining = false;
              runClearTimeout(timeout);
            }
            process.nextTick = function (fun) {
              var args = new Array(arguments.length - 1);
              if (arguments.length > 1) {
                for (var i = 1; i < arguments.length; i++) {
                  args[i - 1] = arguments[i];
                }
              }
              queue.push(new Item(fun, args));
              if (queue.length === 1 && !draining) {
                runTimeout(drainQueue);
              }
            };

            // v8 likes predictible objects
            function Item(fun, array) {
              this.fun = fun;
              this.array = array;
            }
            Item.prototype.run = function () {
              this.fun.apply(null, this.array);
            };
            process.title = 'browser';
            process.browser = true;
            process.env = {};
            process.argv = [];
            process.version = ''; // empty string to avoid regexp issues
            process.versions = {};
            function noop() {}
            process.on = noop;
            process.addListener = noop;
            process.once = noop;
            process.off = noop;
            process.removeListener = noop;
            process.removeAllListeners = noop;
            process.emit = noop;
            process.prependListener = noop;
            process.prependOnceListener = noop;
            process.listeners = function (name) {
              return [];
            };
            process.binding = function (name) {
              throw new Error('process.binding is not supported');
            };
            process.cwd = function () {
              return '/';
            };
            process.chdir = function (dir) {
              throw new Error('process.chdir is not supported');
            };
            process.umask = function () {
              return 0;
            };
          }, {}],
          15: [function (require, module, exports) {
            var unparse = require('escodegen').generate;
            module.exports = function (ast, vars) {
              if (!vars) vars = {};
              var FAIL = {};
              var result = function walk(node, scopeVars) {
                if (node.type === 'Literal') {
                  return node.value;
                } else if (node.type === 'UnaryExpression') {
                  var val = walk(node.argument);
                  if (node.operator === '+') return +val;
                  if (node.operator === '-') return -val;
                  if (node.operator === '~') return ~val;
                  if (node.operator === '!') return !val;
                  return FAIL;
                } else if (node.type === 'ArrayExpression') {
                  var xs = [];
                  for (var i = 0, l = node.elements.length; i < l; i++) {
                    var x = walk(node.elements[i]);
                    if (x === FAIL) return FAIL;
                    xs.push(x);
                  }
                  return xs;
                } else if (node.type === 'ObjectExpression') {
                  var obj = {};
                  for (var i = 0; i < node.properties.length; i++) {
                    var prop = node.properties[i];
                    var value = prop.value === null ? prop.value : walk(prop.value);
                    if (value === FAIL) return FAIL;
                    obj[prop.key.value || prop.key.name] = value;
                  }
                  return obj;
                } else if (node.type === 'BinaryExpression' || node.type === 'LogicalExpression') {
                  var l = walk(node.left);
                  if (l === FAIL) return FAIL;
                  var r = walk(node.right);
                  if (r === FAIL) return FAIL;
                  var op = node.operator;
                  if (op === '==') return l == r;
                  if (op === '===') return l === r;
                  if (op === '!=') return l != r;
                  if (op === '!==') return l !== r;
                  if (op === '+') return l + r;
                  if (op === '-') return l - r;
                  if (op === '*') return l * r;
                  if (op === '/') return l / r;
                  if (op === '%') return l % r;
                  if (op === '<') return l < r;
                  if (op === '<=') return l <= r;
                  if (op === '>') return l > r;
                  if (op === '>=') return l >= r;
                  if (op === '|') return l | r;
                  if (op === '&') return l & r;
                  if (op === '^') return l ^ r;
                  if (op === '&&') return l && r;
                  if (op === '||') return l || r;
                  return FAIL;
                } else if (node.type === 'Identifier') {
                  if ({}.hasOwnProperty.call(vars, node.name)) {
                    return vars[node.name];
                  } else return FAIL;
                } else if (node.type === 'ThisExpression') {
                  if ({}.hasOwnProperty.call(vars, 'this')) {
                    return vars['this'];
                  } else return FAIL;
                } else if (node.type === 'CallExpression') {
                  var callee = walk(node.callee);
                  if (callee === FAIL) return FAIL;
                  if (typeof callee !== 'function') return FAIL;
                  var ctx = node.callee.object ? walk(node.callee.object) : FAIL;
                  if (ctx === FAIL) ctx = null;
                  var args = [];
                  for (var i = 0, l = node.arguments.length; i < l; i++) {
                    var x = walk(node.arguments[i]);
                    if (x === FAIL) return FAIL;
                    args.push(x);
                  }
                  return callee.apply(ctx, args);
                } else if (node.type === 'MemberExpression') {
                  var obj = walk(node.object);
                  // do not allow access to methods on Function 
                  if (obj === FAIL || typeof obj == 'function') {
                    return FAIL;
                  }
                  if (node.property.type === 'Identifier') {
                    return obj[node.property.name];
                  }
                  var prop = walk(node.property);
                  if (prop === FAIL) return FAIL;
                  return obj[prop];
                } else if (node.type === 'ConditionalExpression') {
                  var val = walk(node.test);
                  if (val === FAIL) return FAIL;
                  return val ? walk(node.consequent) : walk(node.alternate);
                } else if (node.type === 'ExpressionStatement') {
                  var val = walk(node.expression);
                  if (val === FAIL) return FAIL;
                  return val;
                } else if (node.type === 'ReturnStatement') {
                  return walk(node.argument);
                } else if (node.type === 'FunctionExpression') {
                  var bodies = node.body.body;

                  // Create a "scope" for our arguments
                  var oldVars = {};
                  Object.keys(vars).forEach(function (element) {
                    oldVars[element] = vars[element];
                  });
                  for (var i = 0; i < node.params.length; i++) {
                    var key = node.params[i];
                    if (key.type == 'Identifier') {
                      vars[key.name] = null;
                    } else return FAIL;
                  }
                  for (var i in bodies) {
                    if (walk(bodies[i]) === FAIL) {
                      return FAIL;
                    }
                  }
                  // restore the vars and scope after we walk
                  vars = oldVars;
                  var keys = Object.keys(vars);
                  var vals = keys.map(function (key) {
                    return vars[key];
                  });
                  return Function(keys.join(', '), 'return ' + unparse(node)).apply(null, vals);
                } else if (node.type === 'TemplateLiteral') {
                  var str = '';
                  for (var i = 0; i < node.expressions.length; i++) {
                    str += walk(node.quasis[i]);
                    str += walk(node.expressions[i]);
                  }
                  str += walk(node.quasis[i]);
                  return str;
                } else if (node.type === 'TaggedTemplateExpression') {
                  var tag = walk(node.tag);
                  var quasi = node.quasi;
                  var strings = quasi.quasis.map(walk);
                  var values = quasi.expressions.map(walk);
                  return tag.apply(null, [strings].concat(values));
                } else if (node.type === 'TemplateElement') {
                  return node.value.cooked;
                } else return FAIL;
              }(ast);
              return result === FAIL ? undefined : result;
            };
          }, {
            "escodegen": 12
          }],
          "jsonpath": [function (require, module, exports) {
            module.exports = require('./lib/index');
          }, {
            "./lib/index": 5
          }]
        }, {}, ["jsonpath"])("jsonpath");
      });
    })(jsonpath);
    var jsonpathExports = jsonpath.exports;
    var jp = /*@__PURE__*/getDefaultExportFromCjs(jsonpathExports);

    /**
     * Returns the computed data section of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data section
     */
    function getComputedDataSection(data) {
      const computedData = jp.query(data, '$.Section[?(@.TOCHeading==="Chemical and Physical Properties")]' + '.Section[?(@.TOCHeading==="Computed Properties")]');
      return computedData;
    }

    /**
     * Returns a computed data section of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @param sectionName Name of the section to be returned
     * @returns Computed Data section
     */
    function getComputedPropertySection(data, sectionName) {
      const returnData = jp.query(data, `$.Section[?(@.TOCHeading==="${sectionName}")].Information[*]`)[0];
      returnData.Description = jp.query(data, `$.Section[?(@.TOCHeading==="${sectionName}")].Description`)[0];
      return returnData;
    }

    /**
     * Returns the number properties of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @param sectionName Name of the section to get the data from
     * @returns Computed Data
     */
    function getNumberProperties(data, sectionName) {
      const output = {
        value: null,
        label: null,
        reference: {
          description: null
        },
        description: null
      };
      try {
        const computationalData = getComputedDataSection(data);
        const section = getComputedPropertySection(computationalData[0], sectionName);
        output.value = section.Value.Number[0];
        output.label = sectionName;
        output.description = section.Description;
        output.reference.description = section.Reference[0];
        if (section.Value.Unit) {
          output.units = section.Value.Unit;
        }
        // eslint-disable-next-line no-empty
      } catch (error) {}
      return output;
    }

    /**
     * Returns the complexity of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns ComputedData
     */
    function getComplexity(data) {
      const complexity = getNumberProperties(data, 'Complexity');
      return complexity;
    }

    /**
     * Returns the formal charge of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getFormalCharge(data) {
      const formalCharge = getNumberProperties(data, 'Formal Charge');
      return formalCharge;
    }

    /**
     * Returns the heavy atom count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getHeavyAtomCount(data) {
      const heavyAtomCount = getNumberProperties(data, 'Heavy Atom Count');
      return heavyAtomCount;
    }

    /**
     * Returns the hydrogen bond acceptor count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getHydrogenBondAcceptorCount(data) {
      const hydrogenBondAcceptorCount = getNumberProperties(data, 'Hydrogen Bond Acceptor Count');
      return hydrogenBondAcceptorCount;
    }

    /**
     * Returns the hydrogen bond donor count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns ComputedData
     */
    function getHydrogenBondDonorCount(data) {
      const hydrogenBondDonorCount = getNumberProperties(data, 'Hydrogen Bond Donor Count');
      return hydrogenBondDonorCount;
    }

    /**
     * Returns the rotable bond count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns ComputedData
     */
    function getRotableBondCount(data) {
      const rotableBondCount = getNumberProperties(data, 'Rotatable Bond Count');
      return rotableBondCount;
    }

    /**
     * Returns the topological polar surface area in angstrom^2
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getTPSA(data) {
      const tpsa = getNumberProperties(data, 'Topological Polar Surface Area');
      return tpsa;
    }

    /**
     * Returns the xLogP3 of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getxLogP3(data) {
      const xLogP3 = getNumberProperties(data, 'XLogP3');
      return xLogP3;
    }

    /**
     * Returns an object of computed data
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Object of computed data
     */
    function getComputedData(data) {
      return {
        hydrogenBondDonorCount: getHydrogenBondDonorCount(data),
        hydrogenBondAcceptorCount: getHydrogenBondAcceptorCount(data),
        rotableBondCount: getRotableBondCount(data),
        heavyAtomCount: getHeavyAtomCount(data),
        formalCharge: getFormalCharge(data),
        complexity: getComplexity(data),
        tpsa: getTPSA(data),
        xLogP3: getxLogP3(data)
      };
    }

    var isMergeableObject = function isMergeableObject(value) {
      return isNonNullObject(value) && !isSpecial(value);
    };
    function isNonNullObject(value) {
      return !!value && typeof value === 'object';
    }
    function isSpecial(value) {
      var stringValue = Object.prototype.toString.call(value);
      return stringValue === '[object RegExp]' || stringValue === '[object Date]' || isReactElement(value);
    }

    // see https://github.com/facebook/react/blob/b5ac963fb791d1298e7f396236383bc955f916c1/src/isomorphic/classic/element/ReactElement.js#L21-L25
    var canUseSymbol = typeof Symbol === 'function' && Symbol.for;
    var REACT_ELEMENT_TYPE = canUseSymbol ? Symbol.for('react.element') : 0xeac7;
    function isReactElement(value) {
      return value.$$typeof === REACT_ELEMENT_TYPE;
    }
    function emptyTarget(val) {
      return Array.isArray(val) ? [] : {};
    }
    function cloneUnlessOtherwiseSpecified(value, options) {
      return options.clone !== false && options.isMergeableObject(value) ? deepmerge(emptyTarget(value), value, options) : value;
    }
    function defaultArrayMerge(target, source, options) {
      return target.concat(source).map(function (element) {
        return cloneUnlessOtherwiseSpecified(element, options);
      });
    }
    function getMergeFunction(key, options) {
      if (!options.customMerge) {
        return deepmerge;
      }
      var customMerge = options.customMerge(key);
      return typeof customMerge === 'function' ? customMerge : deepmerge;
    }
    function getEnumerableOwnPropertySymbols(target) {
      return Object.getOwnPropertySymbols ? Object.getOwnPropertySymbols(target).filter(function (symbol) {
        return Object.propertyIsEnumerable.call(target, symbol);
      }) : [];
    }
    function getKeys(target) {
      return Object.keys(target).concat(getEnumerableOwnPropertySymbols(target));
    }
    function propertyIsOnObject(object, property) {
      try {
        return property in object;
      } catch (_) {
        return false;
      }
    }

    // Protects from prototype poisoning and unexpected merging up the prototype chain.
    function propertyIsUnsafe(target, key) {
      return propertyIsOnObject(target, key) // Properties are safe to merge if they don't exist in the target yet,
      && !(Object.hasOwnProperty.call(target, key) // unsafe if they exist up the prototype chain,
      && Object.propertyIsEnumerable.call(target, key)); // and also unsafe if they're nonenumerable.
    }
    function mergeObject(target, source, options) {
      var destination = {};
      if (options.isMergeableObject(target)) {
        getKeys(target).forEach(function (key) {
          destination[key] = cloneUnlessOtherwiseSpecified(target[key], options);
        });
      }
      getKeys(source).forEach(function (key) {
        if (propertyIsUnsafe(target, key)) {
          return;
        }
        if (propertyIsOnObject(target, key) && options.isMergeableObject(source[key])) {
          destination[key] = getMergeFunction(key, options)(target[key], source[key], options);
        } else {
          destination[key] = cloneUnlessOtherwiseSpecified(source[key], options);
        }
      });
      return destination;
    }
    function deepmerge(target, source, options) {
      options = options || {};
      options.arrayMerge = options.arrayMerge || defaultArrayMerge;
      options.isMergeableObject = options.isMergeableObject || isMergeableObject;
      // cloneUnlessOtherwiseSpecified is added to `options` so that custom arrayMerge()
      // implementations can use it. The caller may not replace it.
      options.cloneUnlessOtherwiseSpecified = cloneUnlessOtherwiseSpecified;
      var sourceIsArray = Array.isArray(source);
      var targetIsArray = Array.isArray(target);
      var sourceAndTargetTypesMatch = sourceIsArray === targetIsArray;
      if (!sourceAndTargetTypesMatch) {
        return cloneUnlessOtherwiseSpecified(source, options);
      } else if (sourceIsArray) {
        return options.arrayMerge(target, source, options);
      } else {
        return mergeObject(target, source, options);
      }
    }
    deepmerge.all = function deepmergeAll(array, options) {
      if (!Array.isArray(array)) {
        throw new Error('first argument should be an array');
      }
      return array.reduce(function (prev, next) {
        return deepmerge(prev, next, options);
      }, {});
    };
    var deepmerge_1 = deepmerge;
    var cjs = deepmerge_1;
    var merge = /*@__PURE__*/getDefaultExportFromCjs(cjs);

    var quantities$1 = {exports: {}};

    /*
    The MIT License (MIT)
    Copyright © 2006-2007 Kevin C. Olbrich
    Copyright © 2010-2016 LIM SAS (http://lim.eu) - Julien Sanchez

    Permission is hereby granted, free of charge, to any person obtaining a copy of
    this software and associated documentation files (the "Software"), to deal in
    the Software without restriction, including without limitation the rights to
    use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
    of the Software, and to permit persons to whom the Software is furnished to do
    so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
    */
    var quantities = quantities$1.exports;
    (function (module, exports) {
      (function (global, factory) {
        module.exports = factory() ;
      })(quantities, function () {

        /**
         * Tests if a value is a string
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a string, false otherwise
         */
        function isString(value) {
          return typeof value === "string" || value instanceof String;
        }

        /*
         * Prefer stricter Number.isFinite if currently supported.
         * To be dropped when ES6 is finalized. Obsolete browsers will
         * have to use ES6 polyfills.
         */
        var isFiniteImpl = Number.isFinite || window.isFinite;
        /**
         * Tests if a value is a number
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a number, false otherwise
         */
        function isNumber(value) {
          // Number.isFinite allows not to consider NaN or '1' as numbers
          return isFiniteImpl(value);
        }

        /*
         * Identity function
         */
        function identity(value) {
          return value;
        }

        /**
         * Returns unique strings from list
         *
         * @param {string[]} strings - array of strings
         *
         *
         * @returns {string[]} a new array of strings without duplicates
         */
        function uniq(strings) {
          var seen = {};
          return strings.filter(function (item) {
            return seen.hasOwnProperty(item) ? false : seen[item] = true;
          });
        }
        function compareArray(array1, array2) {
          if (array2.length !== array1.length) {
            return false;
          }
          for (var i = 0; i < array1.length; i++) {
            if (array2[i].compareArray) {
              if (!array2[i].compareArray(array1[i])) {
                return false;
              }
            }
            if (array2[i] !== array1[i]) {
              return false;
            }
          }
          return true;
        }
        function assign(target, properties) {
          Object.keys(properties).forEach(function (key) {
            target[key] = properties[key];
          });
        }

        /**
         * Safely multiplies numbers while avoiding floating errors
         * like 0.1 * 0.1 => 0.010000000000000002
         *
         * @param {...number} numbers - numbers to multiply
         *
         * @returns {number} result
         */
        function mulSafe() {
          var result = 1,
            decimals = 0;
          for (var i = 0; i < arguments.length; i++) {
            var arg = arguments[i];
            decimals = decimals + getFractional(arg);
            result *= arg;
          }
          return decimals !== 0 ? round(result, decimals) : result;
        }

        /**
         * Safely divides two numbers while avoiding floating errors
         * like 0.3 / 0.05 => 5.999999999999999
         *
         * @returns {number} result
         * @param {number} num Numerator
         * @param {number} den Denominator
         */
        function divSafe(num, den) {
          if (den === 0) {
            throw new Error("Divide by zero");
          }
          var factor = Math.pow(10, getFractional(den));
          var invDen = factor / (factor * den);
          return mulSafe(num, invDen);
        }

        /**
         * Rounds value at the specified number of decimals
         *
         * @param {number} val - value to round
         * @param {number} decimals - number of decimals
         *
         * @returns {number} rounded number
         */
        function round(val, decimals) {
          return Math.round(val * Math.pow(10, decimals)) / Math.pow(10, decimals);
        }
        function getFractional(num) {
          // Check for NaNs or Infinities
          if (!isFinite(num)) {
            return 0;
          }

          // Faster than parsing strings
          // http://jsperf.com/count-decimals/2
          var count = 0;
          while (num % 1 !== 0) {
            num *= 10;
            count++;
          }
          return count;
        }

        /**
         * Custom error type definition
         * @constructor
         */
        function QtyError() {
          var err;
          if (!this) {
            // Allows to instantiate QtyError without new()
            err = Object.create(QtyError.prototype);
            QtyError.apply(err, arguments);
            return err;
          }
          err = Error.apply(this, arguments);
          this.name = "QtyError";
          this.message = err.message;
          this.stack = err.stack;
        }
        QtyError.prototype = Object.create(Error.prototype, {
          constructor: {
            value: QtyError
          }
        });

        /*
         * Throws incompatible units error
         * @param {string} left - units
         * @param {string} right - units incompatible with first argument
         * @throws "Incompatible units" error
         */
        function throwIncompatibleUnits(left, right) {
          throw new QtyError("Incompatible units: " + left + " and " + right);
        }
        var UNITS = {
          /* prefixes */
          "<googol>": [["googol"], 1e100, "prefix"],
          "<kibi>": [["Ki", "Kibi", "kibi"], Math.pow(2, 10), "prefix"],
          "<mebi>": [["Mi", "Mebi", "mebi"], Math.pow(2, 20), "prefix"],
          "<gibi>": [["Gi", "Gibi", "gibi"], Math.pow(2, 30), "prefix"],
          "<tebi>": [["Ti", "Tebi", "tebi"], Math.pow(2, 40), "prefix"],
          "<pebi>": [["Pi", "Pebi", "pebi"], Math.pow(2, 50), "prefix"],
          "<exi>": [["Ei", "Exi", "exi"], Math.pow(2, 60), "prefix"],
          "<zebi>": [["Zi", "Zebi", "zebi"], Math.pow(2, 70), "prefix"],
          "<yebi>": [["Yi", "Yebi", "yebi"], Math.pow(2, 80), "prefix"],
          "<yotta>": [["Y", "Yotta", "yotta"], 1e24, "prefix"],
          "<zetta>": [["Z", "Zetta", "zetta"], 1e21, "prefix"],
          "<exa>": [["E", "Exa", "exa"], 1e18, "prefix"],
          "<peta>": [["P", "Peta", "peta"], 1e15, "prefix"],
          "<tera>": [["T", "Tera", "tera"], 1e12, "prefix"],
          "<giga>": [["G", "Giga", "giga"], 1e9, "prefix"],
          "<mega>": [["M", "Mega", "mega"], 1e6, "prefix"],
          "<kilo>": [["k", "kilo"], 1e3, "prefix"],
          "<hecto>": [["h", "Hecto", "hecto"], 1e2, "prefix"],
          "<deca>": [["da", "Deca", "deca", "deka"], 1e1, "prefix"],
          "<deci>": [["d", "Deci", "deci"], 1e-1, "prefix"],
          "<centi>": [["c", "Centi", "centi"], 1e-2, "prefix"],
          "<milli>": [["m", "Milli", "milli"], 1e-3, "prefix"],
          "<micro>": [["u", "\u03BC" /*µ as greek letter*/, "\u00B5" /*µ as micro sign*/, "Micro", "mc", "micro"], 1e-6, "prefix"],
          "<nano>": [["n", "Nano", "nano"], 1e-9, "prefix"],
          "<pico>": [["p", "Pico", "pico"], 1e-12, "prefix"],
          "<femto>": [["f", "Femto", "femto"], 1e-15, "prefix"],
          "<atto>": [["a", "Atto", "atto"], 1e-18, "prefix"],
          "<zepto>": [["z", "Zepto", "zepto"], 1e-21, "prefix"],
          "<yocto>": [["y", "Yocto", "yocto"], 1e-24, "prefix"],
          "<1>": [["1", "<1>"], 1, ""],
          /* length units */
          "<meter>": [["m", "meter", "meters", "metre", "metres"], 1.0, "length", ["<meter>"]],
          "<inch>": [["in", "inch", "inches", "\""], 0.0254, "length", ["<meter>"]],
          "<foot>": [["ft", "foot", "feet", "'"], 0.3048, "length", ["<meter>"]],
          "<yard>": [["yd", "yard", "yards"], 0.9144, "length", ["<meter>"]],
          "<mile>": [["mi", "mile", "miles"], 1609.344, "length", ["<meter>"]],
          "<naut-mile>": [["nmi", "naut-mile"], 1852, "length", ["<meter>"]],
          "<league>": [["league", "leagues"], 4828, "length", ["<meter>"]],
          "<furlong>": [["furlong", "furlongs"], 201.2, "length", ["<meter>"]],
          "<rod>": [["rd", "rod", "rods"], 5.029, "length", ["<meter>"]],
          "<mil>": [["mil", "mils"], 0.0000254, "length", ["<meter>"]],
          "<angstrom>": [["ang", "angstrom", "angstroms"], 1e-10, "length", ["<meter>"]],
          "<fathom>": [["fathom", "fathoms"], 1.829, "length", ["<meter>"]],
          "<pica>": [["pica", "picas"], 0.00423333333, "length", ["<meter>"]],
          "<point>": [["pt", "point", "points"], 0.000352777778, "length", ["<meter>"]],
          "<redshift>": [["z", "red-shift", "redshift"], 1.302773e26, "length", ["<meter>"]],
          "<AU>": [["AU", "astronomical-unit"], 149597900000, "length", ["<meter>"]],
          "<light-second>": [["ls", "light-second"], 299792500, "length", ["<meter>"]],
          "<light-minute>": [["lmin", "light-minute"], 17987550000, "length", ["<meter>"]],
          "<light-year>": [["ly", "light-year"], 9460528000000000, "length", ["<meter>"]],
          "<parsec>": [["pc", "parsec", "parsecs"], 30856780000000000, "length", ["<meter>"]],
          "<datamile>": [["DM", "datamile"], 1828.8, "length", ["<meter>"]],
          /* mass */
          "<kilogram>": [["kg", "kilogram", "kilograms"], 1.0, "mass", ["<kilogram>"]],
          "<AMU>": [["u", "AMU", "amu"], 1.660538921e-27, "mass", ["<kilogram>"]],
          "<dalton>": [["Da", "Dalton", "Daltons", "dalton", "daltons"], 1.660538921e-27, "mass", ["<kilogram>"]],
          "<slug>": [["slug", "slugs"], 14.5939029, "mass", ["<kilogram>"]],
          "<short-ton>": [["tn", "ton", "short-ton"], 907.18474, "mass", ["<kilogram>"]],
          "<metric-ton>": [["t", "tonne", "metric-ton"], 1000, "mass", ["<kilogram>"]],
          "<carat>": [["ct", "carat", "carats"], 0.0002, "mass", ["<kilogram>"]],
          "<pound>": [["lbs", "lb", "pound", "pounds", "#"], 0.45359237, "mass", ["<kilogram>"]],
          "<ounce>": [["oz", "ounce", "ounces"], 0.0283495231, "mass", ["<kilogram>"]],
          "<gram>": [["g", "gram", "grams", "gramme", "grammes"], 1e-3, "mass", ["<kilogram>"]],
          "<grain>": [["grain", "grains", "gr"], 6.479891e-5, "mass", ["<kilogram>"]],
          "<dram>": [["dram", "drams", "dr"], 0.0017718452, "mass", ["<kilogram>"]],
          "<stone>": [["stone", "stones", "st"], 6.35029318, "mass", ["<kilogram>"]],
          /* area */
          "<hectare>": [["hectare"], 10000, "area", ["<meter>", "<meter>"]],
          "<acre>": [["acre", "acres"], 4046.85642, "area", ["<meter>", "<meter>"]],
          "<sqft>": [["sqft"], 1, "area", ["<foot>", "<foot>"]],
          /* volume */
          "<liter>": [["l", "L", "liter", "liters", "litre", "litres"], 0.001, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<gallon>": [["gal", "gallon", "gallons"], 0.0037854118, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<gallon-imp>": [["galimp", "gallon-imp", "gallons-imp"], 0.0045460900, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<quart>": [["qt", "quart", "quarts"], 0.00094635295, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<pint>": [["pt", "pint", "pints"], 0.000473176475, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<pint-imp>": [["ptimp", "pint-imp", "pints-imp"], 5.6826125e-4, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<cup>": [["cu", "cup", "cups"], 0.000236588238, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<fluid-ounce>": [["floz", "fluid-ounce", "fluid-ounces"], 2.95735297e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<fluid-ounce-imp>": [["flozimp", "floz-imp", "fluid-ounce-imp", "fluid-ounces-imp"], 2.84130625e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<tablespoon>": [["tb", "tbsp", "tbs", "tablespoon", "tablespoons"], 1.47867648e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<teaspoon>": [["tsp", "teaspoon", "teaspoons"], 4.92892161e-6, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<bushel>": [["bu", "bsh", "bushel", "bushels"], 0.035239072, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<oilbarrel>": [["bbl", "oilbarrel", "oilbarrels", "oil-barrel", "oil-barrels"], 0.158987294928, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<beerbarrel>": [["bl", "bl-us", "beerbarrel", "beerbarrels", "beer-barrel", "beer-barrels"], 0.1173477658, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<beerbarrel-imp>": [["blimp", "bl-imp", "beerbarrel-imp", "beerbarrels-imp", "beer-barrel-imp", "beer-barrels-imp"], 0.16365924, "volume", ["<meter>", "<meter>", "<meter>"]],
          /* speed */
          "<kph>": [["kph"], 0.277777778, "speed", ["<meter>"], ["<second>"]],
          "<mph>": [["mph"], 0.44704, "speed", ["<meter>"], ["<second>"]],
          "<knot>": [["kt", "kn", "kts", "knot", "knots"], 0.514444444, "speed", ["<meter>"], ["<second>"]],
          "<fps>": [["fps"], 0.3048, "speed", ["<meter>"], ["<second>"]],
          /* acceleration */
          "<gee>": [["gee"], 9.80665, "acceleration", ["<meter>"], ["<second>", "<second>"]],
          "<Gal>": [["Gal"], 1e-2, "acceleration", ["<meter>"], ["<second>", "<second>"]],
          /* temperature_difference */
          "<kelvin>": [["degK", "kelvin"], 1.0, "temperature", ["<kelvin>"]],
          "<celsius>": [["degC", "celsius", "celsius", "centigrade"], 1.0, "temperature", ["<kelvin>"]],
          "<fahrenheit>": [["degF", "fahrenheit"], 5 / 9, "temperature", ["<kelvin>"]],
          "<rankine>": [["degR", "rankine"], 5 / 9, "temperature", ["<kelvin>"]],
          "<temp-K>": [["tempK", "temp-K"], 1.0, "temperature", ["<temp-K>"]],
          "<temp-C>": [["tempC", "temp-C"], 1.0, "temperature", ["<temp-K>"]],
          "<temp-F>": [["tempF", "temp-F"], 5 / 9, "temperature", ["<temp-K>"]],
          "<temp-R>": [["tempR", "temp-R"], 5 / 9, "temperature", ["<temp-K>"]],
          /* time */
          "<second>": [["s", "sec", "secs", "second", "seconds"], 1.0, "time", ["<second>"]],
          "<minute>": [["min", "mins", "minute", "minutes"], 60.0, "time", ["<second>"]],
          "<hour>": [["h", "hr", "hrs", "hour", "hours"], 3600.0, "time", ["<second>"]],
          "<day>": [["d", "day", "days"], 3600 * 24, "time", ["<second>"]],
          "<week>": [["wk", "week", "weeks"], 7 * 3600 * 24, "time", ["<second>"]],
          "<fortnight>": [["fortnight", "fortnights"], 1209600, "time", ["<second>"]],
          "<year>": [["y", "yr", "year", "years", "annum"], 31556926, "time", ["<second>"]],
          "<decade>": [["decade", "decades"], 315569260, "time", ["<second>"]],
          "<century>": [["century", "centuries"], 3155692600, "time", ["<second>"]],
          /* pressure */
          "<pascal>": [["Pa", "pascal", "Pascal"], 1.0, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<bar>": [["bar", "bars"], 100000, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<mmHg>": [["mmHg"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<inHg>": [["inHg"], 3386.3881472, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<torr>": [["torr"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<atm>": [["atm", "ATM", "atmosphere", "atmospheres"], 101325, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<psi>": [["psi"], 6894.76, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<cmh2o>": [["cmH2O", "cmh2o"], 98.0638, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<inh2o>": [["inH2O", "inh2o"], 249.082052, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          /* viscosity */
          "<poise>": [["P", "poise"], 0.1, "viscosity", ["<kilogram>"], ["<meter>", "<second>"]],
          "<stokes>": [["St", "stokes"], 1e-4, "viscosity", ["<meter>", "<meter>"], ["<second>"]],
          /* substance */
          "<mole>": [["mol", "mole"], 1.0, "substance", ["<mole>"]],
          /* molar_concentration */
          "<molar>": [["M", "molar"], 1000, "molar_concentration", ["<mole>"], ["<meter>", "<meter>", "<meter>"]],
          "<wtpercent>": [["wt%", "wtpercent"], 10, "molar_concentration", ["<kilogram>"], ["<meter>", "<meter>", "<meter>"]],
          /* activity */
          "<katal>": [["kat", "katal", "Katal"], 1.0, "activity", ["<mole>"], ["<second>"]],
          "<unit>": [["U", "enzUnit", "unit"], 16.667e-16, "activity", ["<mole>"], ["<second>"]],
          /* capacitance */
          "<farad>": [["F", "farad", "Farad"], 1.0, "capacitance", ["<second>", "<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<meter>", "<meter>", "<kilogram>"]],
          /* charge */
          "<coulomb>": [["C", "coulomb", "Coulomb"], 1.0, "charge", ["<ampere>", "<second>"]],
          "<Ah>": [["Ah"], 3600, "charge", ["<ampere>", "<second>"]],
          /* current */
          "<ampere>": [["A", "Ampere", "ampere", "amp", "amps"], 1.0, "current", ["<ampere>"]],
          /* conductance */
          "<siemens>": [["S", "Siemens", "siemens"], 1.0, "conductance", ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<kilogram>", "<meter>", "<meter>"]],
          /* inductance */
          "<henry>": [["H", "Henry", "henry"], 1.0, "inductance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>", "<ampere>"]],
          /* potential */
          "<volt>": [["V", "Volt", "volt", "volts"], 1.0, "potential", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>"]],
          /* resistance */
          "<ohm>": [["Ohm", "ohm", "\u03A9" /*Ω as greek letter*/, "\u2126" /*Ω as ohm sign*/], 1.0, "resistance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"]],
          /* magnetism */
          "<weber>": [["Wb", "weber", "webers"], 1.0, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<tesla>": [["T", "tesla", "teslas"], 1.0, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<gauss>": [["G", "gauss"], 1e-4, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<maxwell>": [["Mx", "maxwell", "maxwells"], 1e-8, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<oersted>": [["Oe", "oersted", "oersteds"], 250.0 / Math.PI, "magnetism", ["<ampere>"], ["<meter>"]],
          /* energy */
          "<joule>": [["J", "joule", "Joule", "joules", "Joules"], 1.0, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<erg>": [["erg", "ergs"], 1e-7, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<btu>": [["BTU", "btu", "BTUs"], 1055.056, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<calorie>": [["cal", "calorie", "calories"], 4.18400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<Calorie>": [["Cal", "Calorie", "Calories"], 4184.00, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<therm-US>": [["th", "therm", "therms", "Therm", "therm-US"], 105480400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<Wh>": [["Wh"], 3600, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<electronvolt>": [["eV", "electronvolt", "electronvolts"], 1.602176634E-19, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          /* force */
          "<newton>": [["N", "Newton", "newton"], 1.0, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
          "<dyne>": [["dyn", "dyne"], 1e-5, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
          "<pound-force>": [["lbf", "pound-force"], 4.448222, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
          /* frequency */
          "<hertz>": [["Hz", "hertz", "Hertz"], 1.0, "frequency", ["<1>"], ["<second>"]],
          /* angle */
          "<radian>": [["rad", "radian", "radians"], 1.0, "angle", ["<radian>"]],
          "<degree>": [["deg", "degree", "degrees"], Math.PI / 180.0, "angle", ["<radian>"]],
          "<arcminute>": [["arcmin", "arcminute", "arcminutes"], Math.PI / 10800.0, "angle", ["<radian>"]],
          "<arcsecond>": [["arcsec", "arcsecond", "arcseconds"], Math.PI / 648000.0, "angle", ["<radian>"]],
          "<gradian>": [["gon", "grad", "gradian", "grads"], Math.PI / 200.0, "angle", ["<radian>"]],
          "<steradian>": [["sr", "steradian", "steradians"], 1.0, "solid_angle", ["<steradian>"]],
          /* rotation */
          "<rotation>": [["rotation"], 2.0 * Math.PI, "angle", ["<radian>"]],
          "<rpm>": [["rpm"], 2.0 * Math.PI / 60.0, "angular_velocity", ["<radian>"], ["<second>"]],
          /* information */
          "<byte>": [["B", "byte", "bytes"], 1.0, "information", ["<byte>"]],
          "<bit>": [["b", "bit", "bits"], 0.125, "information", ["<byte>"]],
          /* information rate */
          "<Bps>": [["Bps"], 1.0, "information_rate", ["<byte>"], ["<second>"]],
          "<bps>": [["bps"], 0.125, "information_rate", ["<byte>"], ["<second>"]],
          /* currency */
          "<dollar>": [["USD", "dollar"], 1.0, "currency", ["<dollar>"]],
          "<cents>": [["cents"], 0.01, "currency", ["<dollar>"]],
          /* luminosity */
          "<candela>": [["cd", "candela"], 1.0, "luminosity", ["<candela>"]],
          "<lumen>": [["lm", "lumen"], 1.0, "luminous_power", ["<candela>", "<steradian>"]],
          "<lux>": [["lux"], 1.0, "illuminance", ["<candela>", "<steradian>"], ["<meter>", "<meter>"]],
          /* power */
          "<watt>": [["W", "watt", "watts"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          "<volt-ampere>": [["VA", "volt-ampere"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          "<volt-ampere-reactive>": [["var", "Var", "VAr", "VAR", "volt-ampere-reactive"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          "<horsepower>": [["hp", "horsepower"], 745.699872, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          /* radiation */
          "<gray>": [["Gy", "gray", "grays"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
          "<roentgen>": [["R", "roentgen"], 0.009330, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
          "<sievert>": [["Sv", "sievert", "sieverts"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
          "<becquerel>": [["Bq", "becquerel", "becquerels"], 1.0, "radiation", ["<1>"], ["<second>"]],
          "<curie>": [["Ci", "curie", "curies"], 3.7e10, "radiation", ["<1>"], ["<second>"]],
          /* rate */
          "<cpm>": [["cpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
          "<dpm>": [["dpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
          "<bpm>": [["bpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
          /* resolution / typography */
          "<dot>": [["dot", "dots"], 1, "resolution", ["<each>"]],
          "<pixel>": [["pixel", "px"], 1, "resolution", ["<each>"]],
          "<ppi>": [["ppi"], 1, "resolution", ["<pixel>"], ["<inch>"]],
          "<dpi>": [["dpi"], 1, "typography", ["<dot>"], ["<inch>"]],
          /* other */
          "<cell>": [["cells", "cell"], 1, "counting", ["<each>"]],
          "<each>": [["each"], 1.0, "counting", ["<each>"]],
          "<count>": [["count"], 1.0, "counting", ["<each>"]],
          "<base-pair>": [["bp", "base-pair"], 1.0, "counting", ["<each>"]],
          "<nucleotide>": [["nt", "nucleotide"], 1.0, "counting", ["<each>"]],
          "<molecule>": [["molecule", "molecules"], 1.0, "counting", ["<1>"]],
          "<dozen>": [["doz", "dz", "dozen"], 12.0, "prefix_only", ["<each>"]],
          "<percent>": [["%", "percent"], 0.01, "prefix_only", ["<1>"]],
          "<ppm>": [["ppm"], 1e-6, "prefix_only", ["<1>"]],
          "<ppb>": [["ppb"], 1e-9, "prefix_only", ["<1>"]],
          "<ppt>": [["ppt"], 1e-12, "prefix_only", ["<1>"]],
          "<ppq>": [["ppq"], 1e-15, "prefix_only", ["<1>"]],
          "<gross>": [["gr", "gross"], 144.0, "prefix_only", ["<dozen>", "<dozen>"]],
          "<decibel>": [["dB", "decibel", "decibels"], 1.0, "logarithmic", ["<decibel>"]]
        };
        var BASE_UNITS = ["<meter>", "<kilogram>", "<second>", "<mole>", "<ampere>", "<radian>", "<kelvin>", "<temp-K>", "<byte>", "<dollar>", "<candela>", "<each>", "<steradian>", "<decibel>"];
        var UNITY = "<1>";
        var UNITY_ARRAY = [UNITY];

        // Setup

        /**
         * Asserts unit definition is valid
         *
         * @param {string} unitDef - Name of unit to test
         * @param {Object} definition - Definition of unit to test
         *
         * @returns {void}
         * @throws {QtyError} if unit definition is not valid
         */
        function validateUnitDefinition(unitDef, definition) {
          var scalar = definition[1];
          var numerator = definition[3] || [];
          var denominator = definition[4] || [];
          if (!isNumber(scalar)) {
            throw new QtyError(unitDef + ": Invalid unit definition. " + "'scalar' must be a number");
          }
          numerator.forEach(function (unit) {
            if (UNITS[unit] === undefined) {
              throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'numerator' is not recognized");
            }
          });
          denominator.forEach(function (unit) {
            if (UNITS[unit] === undefined) {
              throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'denominator' is not recognized");
            }
          });
        }
        var PREFIX_VALUES = {};
        var PREFIX_MAP = {};
        var UNIT_VALUES = {};
        var UNIT_MAP = {};
        var OUTPUT_MAP = {};
        for (var unitDef in UNITS) {
          if (UNITS.hasOwnProperty(unitDef)) {
            var definition = UNITS[unitDef];
            if (definition[2] === "prefix") {
              PREFIX_VALUES[unitDef] = definition[1];
              for (var i = 0; i < definition[0].length; i++) {
                PREFIX_MAP[definition[0][i]] = unitDef;
              }
            } else {
              validateUnitDefinition(unitDef, definition);
              UNIT_VALUES[unitDef] = {
                scalar: definition[1],
                numerator: definition[3],
                denominator: definition[4]
              };
              for (var j = 0; j < definition[0].length; j++) {
                UNIT_MAP[definition[0][j]] = unitDef;
              }
            }
            OUTPUT_MAP[unitDef] = definition[0][0];
          }
        }

        /**
         * Returns a list of available units of kind
         *
         * @param {string} [kind] - kind of units
         * @returns {array} names of units
         * @throws {QtyError} if kind is unknown
         */
        function getUnits(kind) {
          var i;
          var units = [];
          var unitKeys = Object.keys(UNITS);
          if (typeof kind === "undefined") {
            for (i = 0; i < unitKeys.length; i++) {
              if (["", "prefix"].indexOf(UNITS[unitKeys[i]][2]) === -1) {
                units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
              }
            }
          } else if (this.getKinds().indexOf(kind) === -1) {
            throw new QtyError("Kind not recognized");
          } else {
            for (i = 0; i < unitKeys.length; i++) {
              if (UNITS[unitKeys[i]][2] === kind) {
                units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
              }
            }
          }
          return units.sort(function (a, b) {
            if (a.toLowerCase() < b.toLowerCase()) {
              return -1;
            }
            if (a.toLowerCase() > b.toLowerCase()) {
              return 1;
            }
            return 0;
          });
        }

        /**
         * Returns a list of alternative names for a unit
         *
         * @param {string} unitName - name of unit
         * @returns {string[]} aliases for unit
         * @throws {QtyError} if unit is unknown
         */
        function getAliases(unitName) {
          if (!UNIT_MAP[unitName]) {
            throw new QtyError("Unit not recognized");
          }
          return UNITS[UNIT_MAP[unitName]][0];
        }
        var SIGNATURE_VECTOR = ["length", "time", "temperature", "mass", "current", "substance", "luminosity", "currency", "information", "angle"];

        /*
        calculates the unit signature id for use in comparing compatible units and simplification
        the signature is based on a simple classification of units and is based on the following publication
         Novak, G.S., Jr. "Conversion of units of measurement", IEEE Transactions on Software Engineering,
        21(8), Aug 1995, pp.651-661
        doi://10.1109/32.403789
        http://ieeexplore.ieee.org/Xplore/login.jsp?url=/iel1/32/9079/00403789.pdf?isnumber=9079&prod=JNL&arnumber=403789&arSt=651&ared=661&arAuthor=Novak%2C+G.S.%2C+Jr.
        */
        function unitSignature() {
          if (this.signature) {
            return this.signature;
          }
          var vector = unitSignatureVector.call(this);
          for (var i = 0; i < vector.length; i++) {
            vector[i] *= Math.pow(20, i);
          }
          return vector.reduce(function (previous, current) {
            return previous + current;
          }, 0);
        }

        // calculates the unit signature vector used by unit_signature
        function unitSignatureVector() {
          if (!this.isBase()) {
            return unitSignatureVector.call(this.toBase());
          }
          var vector = new Array(SIGNATURE_VECTOR.length);
          for (var i = 0; i < vector.length; i++) {
            vector[i] = 0;
          }
          var r, n;
          for (var j = 0; j < this.numerator.length; j++) {
            if (r = UNITS[this.numerator[j]]) {
              n = SIGNATURE_VECTOR.indexOf(r[2]);
              if (n >= 0) {
                vector[n] = vector[n] + 1;
              }
            }
          }
          for (var k = 0; k < this.denominator.length; k++) {
            if (r = UNITS[this.denominator[k]]) {
              n = SIGNATURE_VECTOR.indexOf(r[2]);
              if (n >= 0) {
                vector[n] = vector[n] - 1;
              }
            }
          }
          return vector;
        }
        var SIGN = "[+-]";
        var INTEGER = "\\d+";
        var SIGNED_INTEGER = SIGN + "?" + INTEGER;
        var FRACTION = "\\." + INTEGER;
        var FLOAT = "(?:" + INTEGER + "(?:" + FRACTION + ")?" + ")" + "|" + "(?:" + FRACTION + ")";
        var EXPONENT = "[Ee]" + SIGNED_INTEGER;
        var SCI_NUMBER = "(?:" + FLOAT + ")(?:" + EXPONENT + ")?";
        var SIGNED_NUMBER = SIGN + "?\\s*" + SCI_NUMBER;
        var QTY_STRING = "(" + SIGNED_NUMBER + ")?" + "\\s*([^/]*)(?:\/(.+))?";
        var QTY_STRING_REGEX = new RegExp("^" + QTY_STRING + "$");
        var POWER_OP = "\\^|\\*{2}";
        // Allow unit powers representing scalar, length, area, volume; 4 is for some
        // special case representations in SI base units.
        var SAFE_POWER = "[01234]";
        var TOP_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(-?" + SAFE_POWER + "(?![a-zA-Z]))");
        var BOTTOM_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(" + SAFE_POWER + "(?![a-zA-Z]))");

        /* parse a string into a unit object.
         * Typical formats like :
         * "5.6 kg*m/s^2"
         * "5.6 kg*m*s^-2"
         * "5.6 kilogram*meter*second^-2"
         * "2.2 kPa"
         * "37 degC"
         * "1"  -- creates a unitless constant with value 1
         * "GPa"  -- creates a unit with scalar 1 with units 'GPa'
         * 6'4"  -- recognized as 6 feet + 4 inches
         * 8 lbs 8 oz -- recognized as 8 lbs + 8 ounces
         */
        function parse(val) {
          if (!isString(val)) {
            val = val.toString();
          }
          val = val.trim();
          var result = QTY_STRING_REGEX.exec(val);
          if (!result) {
            throw new QtyError(val + ": Quantity not recognized");
          }
          var scalarMatch = result[1];
          if (scalarMatch) {
            // Allow whitespaces between sign and scalar for loose parsing
            scalarMatch = scalarMatch.replace(/\s/g, "");
            this.scalar = parseFloat(scalarMatch);
          } else {
            this.scalar = 1;
          }
          var top = result[2];
          var bottom = result[3];
          var n, x, nx;
          // TODO DRY me
          while (result = TOP_REGEX.exec(top)) {
            n = parseFloat(result[2]);
            if (isNaN(n)) {
              // Prevents infinite loops
              throw new QtyError("Unit exponent is not a number");
            }
            // Disallow unrecognized unit even if exponent is 0
            if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
              throw new QtyError("Unit not recognized");
            }
            x = result[1] + " ";
            nx = "";
            for (var i = 0; i < Math.abs(n); i++) {
              nx += x;
            }
            if (n >= 0) {
              top = top.replace(result[0], nx);
            } else {
              bottom = bottom ? bottom + nx : nx;
              top = top.replace(result[0], "");
            }
          }
          while (result = BOTTOM_REGEX.exec(bottom)) {
            n = parseFloat(result[2]);
            if (isNaN(n)) {
              // Prevents infinite loops
              throw new QtyError("Unit exponent is not a number");
            }
            // Disallow unrecognized unit even if exponent is 0
            if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
              throw new QtyError("Unit not recognized");
            }
            x = result[1] + " ";
            nx = "";
            for (var j = 0; j < n; j++) {
              nx += x;
            }
            bottom = bottom.replace(result[0], nx);
          }
          if (top) {
            this.numerator = parseUnits(top.trim());
          }
          if (bottom) {
            this.denominator = parseUnits(bottom.trim());
          }
        }
        var PREFIX_REGEX = Object.keys(PREFIX_MAP).sort(function (a, b) {
          return b.length - a.length;
        }).join("|");
        var UNIT_REGEX = Object.keys(UNIT_MAP).sort(function (a, b) {
          return b.length - a.length;
        }).join("|");
        /*
         * Minimal boundary regex to support units with Unicode characters
         * \b only works for ASCII
         */
        var BOUNDARY_REGEX = "\\b|$";
        var UNIT_MATCH = "(" + PREFIX_REGEX + ")??(" + UNIT_REGEX + ")(?:" + BOUNDARY_REGEX + ")";
        var UNIT_TEST_REGEX = new RegExp("^\\s*(" + UNIT_MATCH + "[\\s\\*]*)+$");
        var UNIT_MATCH_REGEX = new RegExp(UNIT_MATCH, "g"); // g flag for multiple occurences
        var parsedUnitsCache = {};
        /**
         * Parses and converts units string to normalized unit array.
         * Result is cached to speed up next calls.
         *
         * @param {string} units Units string
         * @returns {string[]} Array of normalized units
         *
         * @example
         * // Returns ["<second>", "<meter>", "<second>"]
         * parseUnits("s m s");
         *
         */
        function parseUnits(units) {
          var cached = parsedUnitsCache[units];
          if (cached) {
            return cached;
          }
          var unitMatch,
            normalizedUnits = [];

          // Scan
          if (!UNIT_TEST_REGEX.test(units)) {
            throw new QtyError("Unit not recognized");
          }
          while (unitMatch = UNIT_MATCH_REGEX.exec(units)) {
            normalizedUnits.push(unitMatch.slice(1));
          }
          normalizedUnits = normalizedUnits.map(function (item) {
            return PREFIX_MAP[item[0]] ? [PREFIX_MAP[item[0]], UNIT_MAP[item[1]]] : [UNIT_MAP[item[1]]];
          });

          // Flatten and remove null elements
          normalizedUnits = normalizedUnits.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          normalizedUnits = normalizedUnits.filter(function (item) {
            return item;
          });
          parsedUnitsCache[units] = normalizedUnits;
          return normalizedUnits;
        }

        /**
         * Parses a string as a quantity
         * @param {string} value - quantity as text
         * @throws if value is not a string
         * @returns {Qty|null} Parsed quantity or null if unrecognized
         */
        function globalParse(value) {
          if (!isString(value)) {
            throw new QtyError("Argument should be a string");
          }
          try {
            return this(value);
          } catch (e) {
            return null;
          }
        }

        /**
         * Tests if a value is a Qty instance
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a Qty instance, false otherwise
         */
        function isQty(value) {
          return value instanceof Qty;
        }
        function Qty(initValue, initUnits) {
          assertValidConstructorArgs.apply(null, arguments);
          if (!isQty(this)) {
            return new Qty(initValue, initUnits);
          }
          this.scalar = null;
          this.baseScalar = null;
          this.signature = null;
          this._conversionCache = {};
          this.numerator = UNITY_ARRAY;
          this.denominator = UNITY_ARRAY;
          if (isDefinitionObject(initValue)) {
            this.scalar = initValue.scalar;
            this.numerator = initValue.numerator && initValue.numerator.length !== 0 ? initValue.numerator : UNITY_ARRAY;
            this.denominator = initValue.denominator && initValue.denominator.length !== 0 ? initValue.denominator : UNITY_ARRAY;
          } else if (initUnits) {
            parse.call(this, initUnits);
            this.scalar = initValue;
          } else {
            parse.call(this, initValue);
          }

          // math with temperatures is very limited
          if (this.denominator.join("*").indexOf("temp") >= 0) {
            throw new QtyError("Cannot divide with temperatures");
          }
          if (this.numerator.join("*").indexOf("temp") >= 0) {
            if (this.numerator.length > 1) {
              throw new QtyError("Cannot multiply by temperatures");
            }
            if (!compareArray(this.denominator, UNITY_ARRAY)) {
              throw new QtyError("Cannot divide with temperatures");
            }
          }
          this.initValue = initValue;
          updateBaseScalar.call(this);
          if (this.isTemperature() && this.baseScalar < 0) {
            throw new QtyError("Temperatures must not be less than absolute zero");
          }
        }
        Qty.prototype = {
          // Properly set up constructor
          constructor: Qty
        };

        /**
         * Asserts constructor arguments are valid
         *
         * @param {*} value - Value to test
         * @param {string} [units] - Optional units when value is passed as a number
         *
         * @returns {void}
         * @throws {QtyError} if constructor arguments are invalid
         */
        function assertValidConstructorArgs(value, units) {
          if (units) {
            if (!(isNumber(value) && isString(units))) {
              throw new QtyError("Only number accepted as initialization value " + "when units are explicitly provided");
            }
          } else {
            if (!(isString(value) || isNumber(value) || isQty(value) || isDefinitionObject(value))) {
              throw new QtyError("Only string, number or quantity accepted as " + "single initialization value");
            }
          }
        }

        /**
         * Tests if a value is a Qty definition object
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a definition object, false otherwise
         */
        function isDefinitionObject(value) {
          return value && typeof value === "object" && value.hasOwnProperty("scalar");
        }
        function updateBaseScalar() {
          if (this.baseScalar) {
            return this.baseScalar;
          }
          if (this.isBase()) {
            this.baseScalar = this.scalar;
            this.signature = unitSignature.call(this);
          } else {
            var base = this.toBase();
            this.baseScalar = base.scalar;
            this.signature = base.signature;
          }
        }
        var KINDS = {
          "-312078": "elastance",
          "-312058": "resistance",
          "-312038": "inductance",
          "-152058": "potential",
          "-152040": "magnetism",
          "-152038": "magnetism",
          "-7997": "specific_volume",
          "-79": "snap",
          "-59": "jolt",
          "-39": "acceleration",
          "-38": "radiation",
          "-20": "frequency",
          "-19": "speed",
          "-18": "viscosity",
          "-17": "volumetric_flow",
          "-1": "wavenumber",
          "0": "unitless",
          "1": "length",
          "2": "area",
          "3": "volume",
          "20": "time",
          "400": "temperature",
          "7941": "yank",
          "7942": "power",
          "7959": "pressure",
          "7961": "force",
          "7962": "energy",
          "7979": "viscosity",
          "7981": "momentum",
          "7982": "angular_momentum",
          "7997": "density",
          "7998": "area_density",
          "8000": "mass",
          "152020": "radiation_exposure",
          "159999": "magnetism",
          "160000": "current",
          "160020": "charge",
          "312058": "conductance",
          "312078": "capacitance",
          "3199980": "activity",
          "3199997": "molar_concentration",
          "3200000": "substance",
          "63999998": "illuminance",
          "64000000": "luminous_power",
          "1280000000": "currency",
          "25599999980": "information_rate",
          "25600000000": "information",
          "511999999980": "angular_velocity",
          "512000000000": "angle"
        };

        /**
         * Returns the list of available well-known kinds of units, e.g.
         * "radiation" or "length".
         *
         * @returns {string[]} names of kinds of units
         */
        function getKinds() {
          return uniq(Object.keys(KINDS).map(function (knownSignature) {
            return KINDS[knownSignature];
          }));
        }
        Qty.prototype.kind = function () {
          return KINDS[this.signature.toString()];
        };
        assign(Qty.prototype, {
          isDegrees: function () {
            // signature may not have been calculated yet
            return (this.signature === null || this.signature === 400) && this.numerator.length === 1 && compareArray(this.denominator, UNITY_ARRAY) && (this.numerator[0].match(/<temp-[CFRK]>/) || this.numerator[0].match(/<(kelvin|celsius|rankine|fahrenheit)>/));
          },
          isTemperature: function () {
            return this.isDegrees() && this.numerator[0].match(/<temp-[CFRK]>/);
          }
        });
        function subtractTemperatures(lhs, rhs) {
          var lhsUnits = lhs.units();
          var rhsConverted = rhs.to(lhsUnits);
          var dstDegrees = Qty(getDegreeUnits(lhsUnits));
          return Qty({
            "scalar": lhs.scalar - rhsConverted.scalar,
            "numerator": dstDegrees.numerator,
            "denominator": dstDegrees.denominator
          });
        }
        function subtractTempDegrees(temp, deg) {
          var tempDegrees = deg.to(getDegreeUnits(temp.units()));
          return Qty({
            "scalar": temp.scalar - tempDegrees.scalar,
            "numerator": temp.numerator,
            "denominator": temp.denominator
          });
        }
        function addTempDegrees(temp, deg) {
          var tempDegrees = deg.to(getDegreeUnits(temp.units()));
          return Qty({
            "scalar": temp.scalar + tempDegrees.scalar,
            "numerator": temp.numerator,
            "denominator": temp.denominator
          });
        }
        function getDegreeUnits(units) {
          if (units === "tempK") {
            return "degK";
          } else if (units === "tempC") {
            return "degC";
          } else if (units === "tempF") {
            return "degF";
          } else if (units === "tempR") {
            return "degR";
          } else {
            throw new QtyError("Unknown type for temp conversion from: " + units);
          }
        }
        function toDegrees(src, dst) {
          var srcDegK = toDegK(src);
          var dstUnits = dst.units();
          var dstScalar;
          if (dstUnits === "degK") {
            dstScalar = srcDegK.scalar;
          } else if (dstUnits === "degC") {
            dstScalar = srcDegK.scalar;
          } else if (dstUnits === "degF") {
            dstScalar = srcDegK.scalar * 9 / 5;
          } else if (dstUnits === "degR") {
            dstScalar = srcDegK.scalar * 9 / 5;
          } else {
            throw new QtyError("Unknown type for degree conversion to: " + dstUnits);
          }
          return Qty({
            "scalar": dstScalar,
            "numerator": dst.numerator,
            "denominator": dst.denominator
          });
        }
        function toDegK(qty) {
          var units = qty.units();
          var q;
          if (units.match(/(deg)[CFRK]/)) {
            q = qty.baseScalar;
          } else if (units === "tempK") {
            q = qty.scalar;
          } else if (units === "tempC") {
            q = qty.scalar;
          } else if (units === "tempF") {
            q = qty.scalar * 5 / 9;
          } else if (units === "tempR") {
            q = qty.scalar * 5 / 9;
          } else {
            throw new QtyError("Unknown type for temp conversion from: " + units);
          }
          return Qty({
            "scalar": q,
            "numerator": ["<kelvin>"],
            "denominator": UNITY_ARRAY
          });
        }
        function toTemp(src, dst) {
          var dstUnits = dst.units();
          var dstScalar;
          if (dstUnits === "tempK") {
            dstScalar = src.baseScalar;
          } else if (dstUnits === "tempC") {
            dstScalar = src.baseScalar - 273.15;
          } else if (dstUnits === "tempF") {
            dstScalar = src.baseScalar * 9 / 5 - 459.67;
          } else if (dstUnits === "tempR") {
            dstScalar = src.baseScalar * 9 / 5;
          } else {
            throw new QtyError("Unknown type for temp conversion to: " + dstUnits);
          }
          return Qty({
            "scalar": dstScalar,
            "numerator": dst.numerator,
            "denominator": dst.denominator
          });
        }
        function toTempK(qty) {
          var units = qty.units();
          var q;
          if (units.match(/(deg)[CFRK]/)) {
            q = qty.baseScalar;
          } else if (units === "tempK") {
            q = qty.scalar;
          } else if (units === "tempC") {
            q = qty.scalar + 273.15;
          } else if (units === "tempF") {
            q = (qty.scalar + 459.67) * 5 / 9;
          } else if (units === "tempR") {
            q = qty.scalar * 5 / 9;
          } else {
            throw new QtyError("Unknown type for temp conversion from: " + units);
          }
          return Qty({
            "scalar": q,
            "numerator": ["<temp-K>"],
            "denominator": UNITY_ARRAY
          });
        }
        assign(Qty.prototype, {
          /**
           * Converts to other compatible units.
           * Instance's converted quantities are cached for faster subsequent calls.
           *
           * @param {(string|Qty)} other - Target units as string or retrieved from
           *                               other Qty instance (scalar is ignored)
           *
           * @returns {Qty} New converted Qty instance with target units
           *
           * @throws {QtyError} if target units are incompatible
           *
           * @example
           * var weight = Qty("25 kg");
           * weight.to("lb"); // => Qty("55.11556554621939 lbs");
           * weight.to(Qty("3 g")); // => Qty("25000 g"); // scalar of passed Qty is ignored
           */
          to: function (other) {
            var cached, target;
            if (other === undefined || other === null) {
              return this;
            }
            if (!isString(other)) {
              return this.to(other.units());
            }
            cached = this._conversionCache[other];
            if (cached) {
              return cached;
            }

            // Instantiating target to normalize units
            target = Qty(other);
            if (target.units() === this.units()) {
              return this;
            }
            if (!this.isCompatible(target)) {
              if (this.isInverse(target)) {
                target = this.inverse().to(other);
              } else {
                throwIncompatibleUnits(this.units(), target.units());
              }
            } else {
              if (target.isTemperature()) {
                target = toTemp(this, target);
              } else if (target.isDegrees()) {
                target = toDegrees(this, target);
              } else {
                var q = divSafe(this.baseScalar, target.baseScalar);
                target = Qty({
                  "scalar": q,
                  "numerator": target.numerator,
                  "denominator": target.denominator
                });
              }
            }
            this._conversionCache[other] = target;
            return target;
          },
          // convert to base SI units
          // results of the conversion are cached so subsequent calls to this will be fast
          toBase: function () {
            if (this.isBase()) {
              return this;
            }
            if (this.isTemperature()) {
              return toTempK(this);
            }
            var cached = baseUnitCache[this.units()];
            if (!cached) {
              cached = toBaseUnits(this.numerator, this.denominator);
              baseUnitCache[this.units()] = cached;
            }
            return cached.mul(this.scalar);
          },
          // Converts the unit back to a float if it is unitless.  Otherwise raises an exception
          toFloat: function () {
            if (this.isUnitless()) {
              return this.scalar;
            }
            throw new QtyError("Can't convert to Float unless unitless.  Use Unit#scalar");
          },
          /**
           * Returns the nearest multiple of quantity passed as
           * precision
           *
           * @param {(Qty|string|number)} precQuantity - Quantity, string formated
           *   quantity or number as expected precision
           *
           * @returns {Qty} Nearest multiple of precQuantity
           *
           * @example
           * Qty('5.5 ft').toPrec('2 ft'); // returns 6 ft
           * Qty('0.8 cu').toPrec('0.25 cu'); // returns 0.75 cu
           * Qty('6.3782 m').toPrec('cm'); // returns 6.38 m
           * Qty('1.146 MPa').toPrec('0.1 bar'); // returns 1.15 MPa
           *
           */
          toPrec: function (precQuantity) {
            if (isString(precQuantity)) {
              precQuantity = Qty(precQuantity);
            }
            if (isNumber(precQuantity)) {
              precQuantity = Qty(precQuantity + " " + this.units());
            }
            if (!this.isUnitless()) {
              precQuantity = precQuantity.to(this.units());
            } else if (!precQuantity.isUnitless()) {
              throwIncompatibleUnits(this.units(), precQuantity.units());
            }
            if (precQuantity.scalar === 0) {
              throw new QtyError("Divide by zero");
            }
            var precRoundedResult = mulSafe(Math.round(this.scalar / precQuantity.scalar), precQuantity.scalar);
            return Qty(precRoundedResult + this.units());
          }
        });

        /**
         * Configures and returns a fast function to convert
         * Number values from units to others.
         * Useful to efficiently convert large array of values
         * with same units into others with iterative methods.
         * Does not take care of rounding issues.
         *
         * @param {string} srcUnits Units of values to convert
         * @param {string} dstUnits Units to convert to
         *
         * @returns {Function} Converting function accepting Number value
         *   and returning converted value
         *
         * @throws "Incompatible units" if units are incompatible
         *
         * @example
         * // Converting large array of numbers with the same units
         * // into other units
         * var converter = Qty.swiftConverter("m/h", "ft/s");
         * var convertedSerie = largeSerie.map(converter);
         *
         */
        function swiftConverter(srcUnits, dstUnits) {
          var srcQty = Qty(srcUnits);
          var dstQty = Qty(dstUnits);
          if (srcQty.eq(dstQty)) {
            return identity;
          }
          var convert;
          if (!srcQty.isTemperature()) {
            convert = function (value) {
              return value * srcQty.baseScalar / dstQty.baseScalar;
            };
          } else {
            convert = function (value) {
              // TODO Not optimized
              return srcQty.mul(value).to(dstQty).scalar;
            };
          }
          return function converter(value) {
            var i, length, result;
            if (!Array.isArray(value)) {
              return convert(value);
            } else {
              length = value.length;
              result = [];
              for (i = 0; i < length; i++) {
                result.push(convert(value[i]));
              }
              return result;
            }
          };
        }
        var baseUnitCache = {};
        function toBaseUnits(numerator, denominator) {
          var num = [];
          var den = [];
          var q = 1;
          var unit;
          for (var i = 0; i < numerator.length; i++) {
            unit = numerator[i];
            if (PREFIX_VALUES[unit]) {
              // workaround to fix
              // 0.1 * 0.1 => 0.010000000000000002
              q = mulSafe(q, PREFIX_VALUES[unit]);
            } else {
              if (UNIT_VALUES[unit]) {
                q *= UNIT_VALUES[unit].scalar;
                if (UNIT_VALUES[unit].numerator) {
                  num.push(UNIT_VALUES[unit].numerator);
                }
                if (UNIT_VALUES[unit].denominator) {
                  den.push(UNIT_VALUES[unit].denominator);
                }
              }
            }
          }
          for (var j = 0; j < denominator.length; j++) {
            unit = denominator[j];
            if (PREFIX_VALUES[unit]) {
              q /= PREFIX_VALUES[unit];
            } else {
              if (UNIT_VALUES[unit]) {
                q /= UNIT_VALUES[unit].scalar;
                if (UNIT_VALUES[unit].numerator) {
                  den.push(UNIT_VALUES[unit].numerator);
                }
                if (UNIT_VALUES[unit].denominator) {
                  num.push(UNIT_VALUES[unit].denominator);
                }
              }
            }
          }

          // Flatten
          num = num.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          den = den.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          return Qty({
            "scalar": q,
            "numerator": num,
            "denominator": den
          });
        }
        Qty.parse = globalParse;
        Qty.getUnits = getUnits;
        Qty.getAliases = getAliases;
        Qty.mulSafe = mulSafe;
        Qty.divSafe = divSafe;
        Qty.getKinds = getKinds;
        Qty.swiftConverter = swiftConverter;
        Qty.Error = QtyError;
        assign(Qty.prototype, {
          // Returns new instance with units of this
          add: function (other) {
            if (isString(other)) {
              other = Qty(other);
            }
            if (!this.isCompatible(other)) {
              throwIncompatibleUnits(this.units(), other.units());
            }
            if (this.isTemperature() && other.isTemperature()) {
              throw new QtyError("Cannot add two temperatures");
            } else if (this.isTemperature()) {
              return addTempDegrees(this, other);
            } else if (other.isTemperature()) {
              return addTempDegrees(other, this);
            }
            return Qty({
              "scalar": this.scalar + other.to(this).scalar,
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          },
          sub: function (other) {
            if (isString(other)) {
              other = Qty(other);
            }
            if (!this.isCompatible(other)) {
              throwIncompatibleUnits(this.units(), other.units());
            }
            if (this.isTemperature() && other.isTemperature()) {
              return subtractTemperatures(this, other);
            } else if (this.isTemperature()) {
              return subtractTempDegrees(this, other);
            } else if (other.isTemperature()) {
              throw new QtyError("Cannot subtract a temperature from a differential degree unit");
            }
            return Qty({
              "scalar": this.scalar - other.to(this).scalar,
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          },
          mul: function (other) {
            if (isNumber(other)) {
              return Qty({
                "scalar": mulSafe(this.scalar, other),
                "numerator": this.numerator,
                "denominator": this.denominator
              });
            } else if (isString(other)) {
              other = Qty(other);
            }
            if ((this.isTemperature() || other.isTemperature()) && !(this.isUnitless() || other.isUnitless())) {
              throw new QtyError("Cannot multiply by temperatures");
            }

            // Quantities should be multiplied with same units if compatible, with base units else
            var op1 = this;
            var op2 = other;

            // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
            // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
            if (op1.isCompatible(op2) && op1.signature !== 400) {
              op2 = op2.to(op1);
            }
            var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.numerator, op2.denominator);
            return Qty({
              "scalar": mulSafe(op1.scalar, op2.scalar, numdenscale[2]),
              "numerator": numdenscale[0],
              "denominator": numdenscale[1]
            });
          },
          div: function (other) {
            if (isNumber(other)) {
              if (other === 0) {
                throw new QtyError("Divide by zero");
              }
              return Qty({
                "scalar": this.scalar / other,
                "numerator": this.numerator,
                "denominator": this.denominator
              });
            } else if (isString(other)) {
              other = Qty(other);
            }
            if (other.scalar === 0) {
              throw new QtyError("Divide by zero");
            }
            if (other.isTemperature()) {
              throw new QtyError("Cannot divide with temperatures");
            } else if (this.isTemperature() && !other.isUnitless()) {
              throw new QtyError("Cannot divide with temperatures");
            }

            // Quantities should be multiplied with same units if compatible, with base units else
            var op1 = this;
            var op2 = other;

            // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
            // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
            if (op1.isCompatible(op2) && op1.signature !== 400) {
              op2 = op2.to(op1);
            }
            var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.denominator, op2.numerator);
            return Qty({
              "scalar": mulSafe(op1.scalar, numdenscale[2]) / op2.scalar,
              "numerator": numdenscale[0],
              "denominator": numdenscale[1]
            });
          },
          // Returns a Qty that is the inverse of this Qty,
          inverse: function () {
            if (this.isTemperature()) {
              throw new QtyError("Cannot divide with temperatures");
            }
            if (this.scalar === 0) {
              throw new QtyError("Divide by zero");
            }
            return Qty({
              "scalar": 1 / this.scalar,
              "numerator": this.denominator,
              "denominator": this.numerator
            });
          }
        });
        function cleanTerms(num1, den1, num2, den2) {
          function notUnity(val) {
            return val !== UNITY;
          }
          num1 = num1.filter(notUnity);
          num2 = num2.filter(notUnity);
          den1 = den1.filter(notUnity);
          den2 = den2.filter(notUnity);
          var combined = {};
          function combineTerms(terms, direction) {
            var k;
            var prefix;
            var prefixValue;
            for (var i = 0; i < terms.length; i++) {
              if (PREFIX_VALUES[terms[i]]) {
                k = terms[i + 1];
                prefix = terms[i];
                prefixValue = PREFIX_VALUES[prefix];
                i++;
              } else {
                k = terms[i];
                prefix = null;
                prefixValue = 1;
              }
              if (k && k !== UNITY) {
                if (combined[k]) {
                  combined[k][0] += direction;
                  var combinedPrefixValue = combined[k][2] ? PREFIX_VALUES[combined[k][2]] : 1;
                  combined[k][direction === 1 ? 3 : 4] *= divSafe(prefixValue, combinedPrefixValue);
                } else {
                  combined[k] = [direction, k, prefix, 1, 1];
                }
              }
            }
          }
          combineTerms(num1, 1);
          combineTerms(den1, -1);
          combineTerms(num2, 1);
          combineTerms(den2, -1);
          var num = [];
          var den = [];
          var scale = 1;
          for (var prop in combined) {
            if (combined.hasOwnProperty(prop)) {
              var item = combined[prop];
              var n;
              if (item[0] > 0) {
                for (n = 0; n < item[0]; n++) {
                  num.push(item[2] === null ? item[1] : [item[2], item[1]]);
                }
              } else if (item[0] < 0) {
                for (n = 0; n < -item[0]; n++) {
                  den.push(item[2] === null ? item[1] : [item[2], item[1]]);
                }
              }
              scale *= divSafe(item[3], item[4]);
            }
          }
          if (num.length === 0) {
            num = UNITY_ARRAY;
          }
          if (den.length === 0) {
            den = UNITY_ARRAY;
          }

          // Flatten
          num = num.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          den = den.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          return [num, den, scale];
        }
        assign(Qty.prototype, {
          eq: function (other) {
            return this.compareTo(other) === 0;
          },
          lt: function (other) {
            return this.compareTo(other) === -1;
          },
          lte: function (other) {
            return this.eq(other) || this.lt(other);
          },
          gt: function (other) {
            return this.compareTo(other) === 1;
          },
          gte: function (other) {
            return this.eq(other) || this.gt(other);
          },
          // Compare two Qty objects. Throws an exception if they are not of compatible types.
          // Comparisons are done based on the value of the quantity in base SI units.
          //
          // NOTE: We cannot compare inverses as that breaks the general compareTo contract:
          //   if a.compareTo(b) < 0 then b.compareTo(a) > 0
          //   if a.compareTo(b) == 0 then b.compareTo(a) == 0
          //
          //   Since "10S" == ".1ohm" (10 > .1) and "10ohm" == ".1S" (10 > .1)
          //     Qty("10S").inverse().compareTo("10ohm") == -1
          //     Qty("10ohm").inverse().compareTo("10S") == -1
          //
          //   If including inverses in the sort is needed, I suggest writing: Qty.sort(qtyArray,units)
          compareTo: function (other) {
            if (isString(other)) {
              return this.compareTo(Qty(other));
            }
            if (!this.isCompatible(other)) {
              throwIncompatibleUnits(this.units(), other.units());
            }
            if (this.baseScalar < other.baseScalar) {
              return -1;
            } else if (this.baseScalar === other.baseScalar) {
              return 0;
            } else if (this.baseScalar > other.baseScalar) {
              return 1;
            }
          },
          // Return true if quantities and units match
          // Unit("100 cm").same(Unit("100 cm"))  # => true
          // Unit("100 cm").same(Unit("1 m"))     # => false
          same: function (other) {
            return this.scalar === other.scalar && this.units() === other.units();
          }
        });
        assign(Qty.prototype, {
          // returns true if no associated units
          // false, even if the units are "unitless" like 'radians, each, etc'
          isUnitless: function () {
            return [this.numerator, this.denominator].every(function (item) {
              return compareArray(item, UNITY_ARRAY);
            });
          },
          /*
          check to see if units are compatible, but not the scalar part
          this check is done by comparing signatures for performance reasons
          if passed a string, it will create a unit object with the string and then do the comparison
          this permits a syntax like:
          unit =~ "mm"
          if you want to do a regexp on the unit string do this ...
          unit.units =~ /regexp/
          */
          isCompatible: function (other) {
            if (isString(other)) {
              return this.isCompatible(Qty(other));
            }
            if (!isQty(other)) {
              return false;
            }
            if (other.signature !== undefined) {
              return this.signature === other.signature;
            } else {
              return false;
            }
          },
          /*
          check to see if units are inverse of each other, but not the scalar part
          this check is done by comparing signatures for performance reasons
          if passed a string, it will create a unit object with the string and then do the comparison
          this permits a syntax like:
          unit =~ "mm"
          if you want to do a regexp on the unit string do this ...
          unit.units =~ /regexp/
          */
          isInverse: function (other) {
            return this.inverse().isCompatible(other);
          },
          // Returns 'true' if the Unit is represented in base units
          isBase: function () {
            if (this._isBase !== undefined) {
              return this._isBase;
            }
            if (this.isDegrees() && this.numerator[0].match(/<(kelvin|temp-K)>/)) {
              this._isBase = true;
              return this._isBase;
            }
            this.numerator.concat(this.denominator).forEach(function (item) {
              if (item !== UNITY && BASE_UNITS.indexOf(item) === -1) {
                this._isBase = false;
              }
            }, this);
            if (this._isBase === false) {
              return this._isBase;
            }
            this._isBase = true;
            return this._isBase;
          }
        });
        function NestedMap() {}
        NestedMap.prototype.get = function (keys) {
          // Allows to pass key1, key2, ... instead of [key1, key2, ...]
          if (arguments.length > 1) {
            // Slower with Firefox but faster with Chrome than
            // Array.prototype.slice.call(arguments)
            // See http://jsperf.com/array-apply-versus-array-prototype-slice-call
            keys = Array.apply(null, arguments);
          }
          return keys.reduce(function (map, key, index) {
            if (map) {
              var childMap = map[key];
              if (index === keys.length - 1) {
                return childMap ? childMap.data : undefined;
              } else {
                return childMap;
              }
            }
          }, this);
        };
        NestedMap.prototype.set = function (keys, value) {
          if (arguments.length > 2) {
            keys = Array.prototype.slice.call(arguments, 0, -1);
            value = arguments[arguments.length - 1];
          }
          return keys.reduce(function (map, key, index) {
            var childMap = map[key];
            if (childMap === undefined) {
              childMap = map[key] = {};
            }
            if (index === keys.length - 1) {
              childMap.data = value;
              return value;
            } else {
              return childMap;
            }
          }, this);
        };

        /**
         * Default formatter
         *
         * @param {number} scalar - scalar value
         * @param {string} units - units as string
         *
         * @returns {string} formatted result
         */
        function defaultFormatter(scalar, units) {
          return (scalar + " " + units).trim();
        }

        /**
         *
         * Configurable Qty default formatter
         *
         * @type {function}
         *
         * @param {number} scalar
         * @param {string} units
         *
         * @returns {string} formatted result
         */
        Qty.formatter = defaultFormatter;
        assign(Qty.prototype, {
          // returns the 'unit' part of the Unit object without the scalar
          units: function () {
            if (this._units !== undefined) {
              return this._units;
            }
            var numIsUnity = compareArray(this.numerator, UNITY_ARRAY);
            var denIsUnity = compareArray(this.denominator, UNITY_ARRAY);
            if (numIsUnity && denIsUnity) {
              this._units = "";
              return this._units;
            }
            var numUnits = stringifyUnits(this.numerator);
            var denUnits = stringifyUnits(this.denominator);
            this._units = numUnits + (denIsUnity ? "" : "/" + denUnits);
            return this._units;
          },
          /**
           * Stringifies the quantity
           * Deprecation notice: only units parameter is supported.
           *
           * @param {(number|string|Qty)} targetUnitsOrMaxDecimalsOrPrec -
           *                              target units if string,
           *                              max number of decimals if number,
           *                              passed to #toPrec before converting if Qty
           *
           * @param {number=} maxDecimals - Maximum number of decimals of
           *                                formatted output
           *
           * @returns {string} reparseable quantity as string
           */
          toString: function (targetUnitsOrMaxDecimalsOrPrec, maxDecimals) {
            var targetUnits;
            if (isNumber(targetUnitsOrMaxDecimalsOrPrec)) {
              targetUnits = this.units();
              maxDecimals = targetUnitsOrMaxDecimalsOrPrec;
            } else if (isString(targetUnitsOrMaxDecimalsOrPrec)) {
              targetUnits = targetUnitsOrMaxDecimalsOrPrec;
            } else if (isQty(targetUnitsOrMaxDecimalsOrPrec)) {
              return this.toPrec(targetUnitsOrMaxDecimalsOrPrec).toString(maxDecimals);
            }
            var out = this.to(targetUnits);
            var outScalar = maxDecimals !== undefined ? round(out.scalar, maxDecimals) : out.scalar;
            out = (outScalar + " " + out.units()).trim();
            return out;
          },
          /**
           * Format the quantity according to optional passed target units
           * and formatter
           *
           * @param {string} [targetUnits=current units] -
           *                 optional units to convert to before formatting
           *
           * @param {function} [formatter=Qty.formatter] -
           *                   delegates formatting to formatter callback.
           *                   formatter is called back with two parameters (scalar, units)
           *                   and should return formatted result.
           *                   If unspecified, formatting is delegated to default formatter
           *                   set to Qty.formatter
           *
           * @example
           * var roundingAndLocalizingFormatter = function(scalar, units) {
           *   // localize or limit scalar to n max decimals for instance
           *   // return formatted result
           * };
           * var qty = Qty('1.1234 m');
           * qty.format(); // same units, default formatter => "1.234 m"
           * qty.format("cm"); // converted to "cm", default formatter => "123.45 cm"
           * qty.format(roundingAndLocalizingFormatter); // same units, custom formatter => "1,2 m"
           * qty.format("cm", roundingAndLocalizingFormatter); // convert to "cm", custom formatter => "123,4 cm"
           *
           * @returns {string} quantity as string
           */
          format: function (targetUnits, formatter) {
            if (arguments.length === 1) {
              if (typeof targetUnits === "function") {
                formatter = targetUnits;
                targetUnits = undefined;
              }
            }
            formatter = formatter || Qty.formatter;
            var targetQty = this.to(targetUnits);
            return formatter.call(this, targetQty.scalar, targetQty.units());
          }
        });
        var stringifiedUnitsCache = new NestedMap();
        /**
         * Returns a string representing a normalized unit array
         *
         * @param {string[]} units Normalized unit array
         * @returns {string} String representing passed normalized unit array and
         *   suitable for output
         *
         */
        function stringifyUnits(units) {
          var stringified = stringifiedUnitsCache.get(units);
          if (stringified) {
            return stringified;
          }
          var isUnity = compareArray(units, UNITY_ARRAY);
          if (isUnity) {
            stringified = "1";
          } else {
            stringified = simplify(getOutputNames(units)).join("*");
          }

          // Cache result
          stringifiedUnitsCache.set(units, stringified);
          return stringified;
        }
        function getOutputNames(units) {
          var unitNames = [],
            token,
            tokenNext;
          for (var i = 0; i < units.length; i++) {
            token = units[i];
            tokenNext = units[i + 1];
            if (PREFIX_VALUES[token]) {
              unitNames.push(OUTPUT_MAP[token] + OUTPUT_MAP[tokenNext]);
              i++;
            } else {
              unitNames.push(OUTPUT_MAP[token]);
            }
          }
          return unitNames;
        }
        function simplify(units) {
          // this turns ['s','m','s'] into ['s2','m']

          var unitCounts = units.reduce(function (acc, unit) {
            var unitCounter = acc[unit];
            if (!unitCounter) {
              acc.push(unitCounter = acc[unit] = [unit, 0]);
            }
            unitCounter[1]++;
            return acc;
          }, []);
          return unitCounts.map(function (unitCount) {
            return unitCount[0] + (unitCount[1] > 1 ? unitCount[1] : "");
          });
        }
        Qty.version = "1.8.0";
        return Qty;
      });
    })(quantities$1);
    var quantitiesExports = quantities$1.exports;
    var Qty = /*@__PURE__*/getDefaultExportFromCjs(quantitiesExports);

    /**
     * Parse a string that may contain min / max and units values
     * @param {string} string
     * @param {object} [options={}]
     * @param {number} [options.defaultValue]
     * @param {string} [options.defjaultUnits]
     * @param {string} [options.targetUnits]
     * @param {string} [options.optional=false] Is this value optional. Would through an error if empty and not optional
     */
    function parseNumbersUnits(string = '', options = {}) {
      const {
        defaultUnits = undefined,
        targetUnits = undefined,
        defaultValue = undefined,
        optional = false
      } = options;
      let normalized = String(string).replace(/ /g, '').replace(/,/g, '.');
      normalized = normalized.replace(/Â°/g, '°');
      if (normalized.match(/^\(.*\)$/)) {
        normalized = normalized.substring(1, normalized.length - 2);
      }

      // still parenthesis ??? We remove what is after
      normalized = normalized.replace(/\(.*/, '');
      if (optional && !normalized) return {};
      let splitter = /^([0-9.Ee-]*)([a-zA-Z^/ °0-9]*)$/;
      if (!normalized.match(splitter)) {
        throw Error(`Can not parseNumbersUnits of: ${string}`);
      }
      let numbers = normalized.replace(splitter, '$1');
      let units = normalized.replace(splitter, '$2').trim();
      let low;
      let high;
      if (numbers === '') {
        if (defaultValue) {
          numbers = String(defaultValue).replace(/ /g, '');
        } else {
          throw Error(`Can not parseNumbersUnits of: ${string}`);
        }
      }
      if (numbers.includes('--')) {
        // -5--4
        low = Number(numbers.split('--')[0]);
        high = Number(`-${numbers.split('--')[1]}`);
      } else if (numbers.startsWith('-')) {
        // -5 or -1-1 (minus 1 to plus 1)
        let subparts = numbers.split('-');
        switch (subparts.length) {
          case 2:
            low = Number(numbers);
            break;
          case 3:
            low = Number(`-${subparts[1]}`);
            high = Number(+subparts[2]);
            break;
          default:
            throw new Error(`Can not parse: ${string}`);
        }
      } else {
        let subparts = numbers.split('-');
        low = subparts[0] !== '' ? Number(subparts[0]) : undefined;
        high = subparts[1] !== undefined ? Number(subparts[1]) : undefined;
      }
      if (!units) units = defaultUnits;

      // Should we convert the units
      if (targetUnits) {
        let convert = Qty.swiftConverter(units.replace(/°/g, 'temp'), targetUnits.replace(/°/g, 'temp'));
        if (low !== undefined) {
          low = convert(low);
        }
        if (high !== undefined) {
          high = convert(high);
        }
        units = targetUnits;
      }
      return {
        low,
        high,
        units
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.temperature={}]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     * @param {object} [options.pressure={}]
     * @param {number} [options.pressure.defaultValue]
     * @param {string} [options.pressure.defaultUnits]
     * @param {string} [options.pressure.targetUnits]
     * @param {string} [options.pressure.optional]
     */
    function parseBoilingPoint(string, options = {}) {
      let parts = String(string).split(/[@/]|, | at /);
      return {
        temperature: parseNumbersUnits(parts[0], options.temperature),
        pressure: parseNumbersUnits(parts[1], options.pressure)
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.value={}]
     * @param {string} [options.value.defaultUnits]
     * @param {string} [options.value.targetUnits]
     * @param {object} [options.temperature={}]
     * @param {number} [options.temperature.defaultValue]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     */
    function parseDensity(string, options = {}) {
      string = String(string);
      string = string.replace(/\(.*/, '');
      string = string.replace('g/cu cm', 'g/cm^3');
      let parts = String(string).split(/[@]|, | at /);
      let value = parts[0];
      let temperature = parts[1];
      if (temperature) {
        temperature = temperature.replace(/ *t= */, '');
      }
      return {
        value: parseNumbersUnits(value, options.value),
        temperature: parseNumbersUnits(temperature, options.temperature)
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.value={}]
     * @param {string} [options.value.defaultUnits]
     * @param {string} [options.value.targetUnits]
     * @param {object} [options.temperature={}]
     * @param {number} [options.temperature.defaultValue]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     */
    function parseRefractiveIndex(string, options = {}) {
      string = String(string).replace(/^[^0-9.]*/, '');
      let parts = string.split(/at|[@/]|, /);
      let value = parts[0];
      let temperature = parts[1];
      if (temperature) {
        temperature = temperature.replace(/ *t= */, '');
      }
      return {
        value: parseNumbersUnits(value, options.value),
        temperature: parseNumbersUnits(temperature, options.temperature)
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.temperature={}]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     * @param {string} [options.pressure.optional]
     * @param {object} [options.pressure={}]
     * @param {string} [options.pressure.defaultUnits]
     * @param {string} [options.pressure.targetUnits]

     */
    function parseVaporPressure(string, options = {}) {
      let parts = String(string).split(/[@/]|, | at /);
      return {
        pressure: parseNumbersUnits(parts[0], options.pressure),
        temperature: parseNumbersUnits(parts[1], options.temperature)
      };
    }

    /**
     * Returns all references cited in the PubChem entry
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns References
     */
    function getReferences(data) {
      const references = jp.query(data, '$.Reference[*]').reduce((ref, entry) => {
        ref[entry.ReferenceNumber] = {
          url: entry.URL,
          sourceName: entry.SourceName,
          name: entry.Name,
          description: entry.Description
        };
        return ref;
      }, {});
      return references;
    }

    function extractExperimentalData(data, sectionName, options = {}) {
      const {
        parser
      } = options;
      const references = getReferences(data);
      const experimentalSection = getExperimentalDataSection(data);
      const results = [];
      if (!experimentalSection || experimentalSection.length === 0) {
        return results;
      }
      jp.query(experimentalSection[0], `$.Section[?(@.TOCHeading==="${sectionName}")].Information[*]`).forEach(entry => {
        jp.query(entry, '$.Value.StringWithMarkup[*].String').forEach(original => {
          const result = {
            reference: references[entry.ReferenceNumber],
            data: {
              original
            }
          };
          if (parser) {
            try {
              result.data.parsed = parser(original, options);
            } catch (e) {
              result.data.parsed = {
                error: e.toString()
              };
            }
          }
          results.push(result);
        });
      });
      return results;
    }
    function getExperimentalDataSection(data) {
      const experimentalData = jp.query(data, '$.Section[?(@.TOCHeading==="Chemical and Physical Properties")]' + '.Section[?(@.TOCHeading==="Experimental Properties")]');
      return experimentalData;
    }

    /**
     * Returns the boiling point of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     * @returns ExperimentalData
     */
    function getBoilingPoint(data, options = {}) {
      const boilingPoint = extractExperimentalData(data, 'Boiling Point', {
        parser: value => parseBoilingPoint(value, merge({
          pressure: {
            defaultValue: 760,
            defaultUnits: 'torr'
          }
        }, options))
      });
      return boilingPoint;
    }

    /**
     * Returns the refractive index of a compound
     * @param data Data of a compound data request to the PubChem API
     * @returns ExperimentalData
     */
    function getDensity(data = {}, options = {}) {
      const density = extractExperimentalData(data, 'Density', {
        parser: value => {
          return parseDensity(value, options);
        }
      });
      return density;
    }

    /**
     *
     *
     * @export
     * @param {Object} data response of a compound data request to the PubChem API
     * @param {object} [options={}]
     * @param {object} [options.targetUnits]
     * @returns {ExperimentalData}
     */
    function getFlashPoint(data, options = {}) {
      const flashPoint = extractExperimentalData(data, 'Flash Point', {
        parser: value => parseNumbersUnits(value, options.temperature)
      });
      return flashPoint;
    }

    /**
     * Returns the melting point of a compound
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     * @returns ExperimentalData
     */
    function getMeltingPoint(data, options = {}) {
      const meltingPoint = extractExperimentalData(data, 'Melting Point', {
        parser: value => parseNumbersUnits(value, options.temperature)
      });
      return meltingPoint;
    }

    /**
     * Returns the refractive index of a compound
     * @param data Data of a compound data request to the PubChem API
     * @returns ExperimentalData
     */
    function getRefractiveIndex(data = {}) {
      const refractiveIndex = extractExperimentalData(data, 'Refractive Index', {
        parser: value => parseRefractiveIndex(value)
      });
      return refractiveIndex;
    }

    /**
     * Returns the solubility of a compound
     * @param data Data of a compound data request to the PubChem API
     * @returns ExperimentalData
     */
    function getSolubility(data) {
      const solubility = extractExperimentalData(data, 'Solubility');
      return solubility;
    }

    /**
     * Returns the vapor pressure of a compound
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     */
    function getVaporPressure(data, options = {}) {
      const vaporPressure = extractExperimentalData(data, 'Vapor Pressure', {
        parser: value => parseVaporPressure(value, merge({
          temperature: {
            optional: true
          }
        }, options))
      });
      return vaporPressure;
    }

    /**
     * Returns the experimental data of a compound
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     * @returns ExperimentalData
     */
    function getExperimentalData(data, options) {
      return {
        boilingPoint: getBoilingPoint(data, options),
        density: getDensity(data, options),
        flashPoint: getFlashPoint(data, options),
        meltingPoint: getMeltingPoint(data, options),
        solubility: getSolubility(data),
        vaporPressure: getVaporPressure(data, options),
        refractiveIndex: getRefractiveIndex(data)
      };
    }

    function getFormula(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Molecular Formula")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'Molecular Formula',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Molecular Formula")]')[0].Description
      };
    }

    // Names and Identifiers
    function getInChI(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChI")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'InChI',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChI")]')[0].Description
      };
    }

    // Names and Identifiers
    function getInChIKey(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChIKey")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'InChIKey',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChIKey")]')[0].Description
      };
    }

    // Names and Identifiers
    function getSMILES(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="Canonical SMILES")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'Canonical SMILES',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="Canonical SMILES")]')[0].Description
      };
    }

    function getIdentifiers(data) {
      return {
        formula: getFormula(data),
        inchi: getInChI(data),
        inchiKey: getInChIKey(data),
        smiles: getSMILES(data)
      };
    }

    const ghsPictogramText = {
      GHS01: 'Explosive',
      GHS02: 'Flammable',
      GHS03: 'Oxidizing',
      GHS04: 'Compressed Gas',
      GHS05: 'Corrosive',
      GHS06: 'Toxic',
      GHS07: 'Harmful Irritant',
      GHS08: 'Health Hazard',
      GHS09: 'Environmental Hazard'
    };

    const hazardStatements = {
      H200: '(Deleted) Unstable Explosive',
      H201: '(Deleted) Explosive; mass explosion hazard',
      H202: '(Deleted) Explosive; severe projection hazard',
      H203: '(Deleted) Explosive; fire, blast or projection hazard',
      H204: 'Fire or projection hazard',
      H205: '(Deleted) May mass explode in fire',
      H206: 'Fire, blast or projection hazard; increased risk of explosion if desensitizing agent is reduced',
      H207: 'Fire or projection hazard; increased risk of explosion if desensitizing agent is reduced',
      H208: 'Fire hazard; increased risk of explosion if desensitizing agent is reduced',
      H209: 'Explosive',
      H210: 'Very sensitive',
      H211: 'Maybe sensitive',
      H220: 'Extremely flammable gas',
      H221: 'Flammable gas',
      H222: 'Extremely flammable aerosol',
      H223: 'Flammable aerosol',
      H224: 'Extremely flammable liquid and vapor',
      H225: 'Highly Flammable liquid and vapor',
      H226: 'Flammable liquid and vapor',
      H227: 'Combustible liquid',
      H228: 'Flammable solid',
      H229: 'Pressurized container: may burst if heated',
      H230: 'May react explosively even in the absence of air',
      H231: 'May react explosively even in the absence of air at elevated pressure and/or temperature',
      H232: 'May ignite spontaneously if exposed to air',
      H240: 'Heating may cause an explosion',
      H241: 'Heating may cause a fire or explosion',
      H242: 'Heating may cause a fire',
      H250: 'Catches fire spontaneously if exposed to air',
      H251: 'Self-heating; may catch fire',
      H252: 'Self-heating in large quantities; may catch fire',
      H260: 'In contact with water releases flammable gases which may ignite spontaneously',
      H261: 'In contact with water releases flammable gas',
      H270: 'May cause or intensify fire; oxidizer',
      H271: 'May cause fire or explosion; strong Oxidizer',
      H272: 'May intensify fire; oxidizer',
      H280: 'Contains gas under pressure; may explode if heated',
      H281: 'Contains refrigerated gas; may cause cryogenic burns or injury',
      H282: 'Extremely flammable chemical under pressure: may explode if heated',
      H283: 'Flammable chemical under pressure: may explode if heated',
      H284: 'Chemical under pressure: may explode if heated',
      H290: 'May be corrosive to metals',
      H300: 'Fatal if swallowed',
      H301: 'Toxic if swallowed',
      H302: 'Harmful if swallowed',
      H303: 'May be harmful if swallowed',
      H304: 'May be fatal if swallowed and enters airways',
      H305: 'May be harmful if swallowed and enters airways',
      H310: 'Fatal in contact with skin',
      H311: 'Toxic in contact with skin',
      H312: 'Harmful in contact with skin',
      H313: 'May be harmful in contact with skin',
      H314: 'Causes severe skin burns and eye damage',
      H315: 'Causes skin irritation',
      H316: 'Causes mild skin irritation',
      H317: 'May cause an allergic skin reaction',
      H318: 'Causes serious eye damage',
      H319: 'Causes serious eye irritation',
      H320: 'Causes eye irritation',
      H330: 'Fatal if inhaled',
      H331: 'Toxic if inhaled',
      H332: 'Harmful if inhaled',
      H333: 'May be harmful if inhaled',
      H334: 'May cause allergy or asthma symptoms or breathing difficulties if inhaled',
      H335: 'May cause respiratory irritation',
      H336: 'May cause drowsiness or dizziness',
      H340: 'May cause genetic defects',
      H341: 'Suspected of causing genetic defects',
      H350: 'May cause cancer',
      H350i: 'May cause cancer by inhalation',
      H351: 'Suspected of causing cancer',
      H360: 'May damage fertility or the unborn child',
      H360F: 'May damage fertility',
      H360D: 'May damage the unborn child',
      H360FD: 'May damage fertility; May damage the unborn child',
      H360Fd: 'May damage fertility; Suspected of damaging the unborn child',
      H360Df: 'May damage the unborn child; Suspected of damaging fertility',
      H361: 'Suspected of damaging fertility or the unborn child',
      H361f: 'Suspected of damaging fertility',
      H361d: 'Suspected of damaging the unborn child',
      H361fd: 'Suspected of damaging fertility; Suspected of damaging the unborn child',
      H362: 'May cause harm to breast-fed children',
      H370: 'Causes damage to organs',
      H371: 'May cause damage to organs',
      H372: 'Causes damage to organs through prolonged or repeated exposure',
      H373: 'May causes damage to organs through prolonged or repeated exposure',
      H400: 'Very toxic to aquatic life',
      H401: 'Toxic to aquatic life',
      H402: 'Harmful to aquatic life',
      H410: 'Very toxic to aquatic life with long lasting effects',
      H411: 'Toxic to aquatic life with long lasting effects',
      H412: 'Harmful to aquatic life with long lasting effects',
      H413: 'May cause long lasting harmful effects to aquatic life',
      H420: 'Harms public health and the environment by destroying ozone in the upper atmosphere',
      'H300+H310': 'Fatal if swallowed or in contact with skin',
      'H300+H330': 'Fatal if swallowed or if inhaled',
      'H310+H330': 'Fatal in contact with skin or if inhaled',
      'H300+H310+H330': 'Fatal if swallowed, in contact with skin or if inhaled',
      'H301+H311': 'Toxic if swallowed or in contact with skin',
      'H301+H331': 'Toxic if swallowed or if inhaled',
      'H311+H331': 'Toxic in contact with skin or if inhaled.',
      'H301+H311+H331': 'Toxic if swallowed, in contact with skin or if inhaled',
      'H302+H312': 'Harmful if swallowed or in contact with skin',
      'H302+H332': 'Harmful if swallowed or if inhaled',
      'H312+H332': 'Harmful in contact with skin or if inhaled',
      'H302+H312+H332': 'Harmful if swallowed, in contact with skin or if inhaled',
      'H303+H313': 'May be harmful if swallowed or in contact with skin',
      'H303+H333': 'May be harmful if swallowed or if inhaled',
      'H313+H333': 'May be harmful in contact with skin or if inhaled',
      'H303+H313+H333': 'May be harmful if swallowed, in contact with skin or if inhaled',
      'H315+H320': 'Cause skin and eye irritation',
      EUH001: 'Explosive when dry',
      EUH006: 'Explosive with or without contact with air',
      EUH014: 'Reacts violently with water',
      EUH018: 'In use may form flammable/explosive vapour-air mixture',
      EUH019: 'May form explosive peroxides',
      EUH044: 'Risk of explosion if heated under confinement',
      EUH029: 'Contact with water liberates toxic gas',
      EUH031: 'Contact with acids liberates toxic gas',
      EUH032: 'Contact with acids liberates very toxic gas',
      EUH066: 'Repeated exposure may cause skin dryness or cracking',
      EUH070: 'Toxic by eye contact',
      EUH071: 'Corrosive to the respiratory tract',
      EUH059: 'Hazardous to the ozone layer, superseded by GHS Class 5.1 in the second adaptation to technical progress of CLP.',
      EUH201: 'Contains lead. Should not be used on surfaces liable to be chewed or sucked by children.',
      EUH201A: 'Warning! Contains lead.',
      EUH202: 'Cyanoacrylate. Danger. Bonds skin and eyes in seconds. Keep out of the reach of children.',
      EUH203: 'Contains chromium(VI). May produce an allergic reaction.',
      EUH204: 'Contains isocyanates. May produce an allergic reaction.',
      EUH205: 'Contains epoxy constituents. May produce an allergic reaction.',
      EUH206: 'Warning! Do not use together with other products. May release dangerous gases (chlorine).',
      EUH207: 'Warning! Contains cadmium. Dangerous fumes are formed during use. See information supplied by the manufacturer. Comply with the safety instructions.',
      EUH208: 'Contains <name of sensitising substance>. May produce an allergic reaction.',
      EUH209: 'Can become highly flammable in use.',
      EUH209A: 'Can become flammable in use.',
      EUH210: 'Safety data sheet available on request.',
      EUH401: 'To avoid risks to human health and the environment, comply with the instructions for use.'
    };

    const precautionaryStatements = {
      P101: 'If medical advice is needed, have product container or label at hand.',
      P102: 'Keep out of reach of children.',
      P103: 'Read carefully and follow all instructions.',
      P201: '(Deleted) Obtain special instructions before use.',
      P202: '(Deleted) Do not handle until all safety precautions have been read and understood.',
      P203: 'Obtain, read and follow all safety instructions before use.',
      P210: 'Keep away from heat, hot surface, sparks, open flames and other ignition sources. No smoking.',
      P211: 'Do not spray on an open flame or other ignition source.',
      P212: 'Avoid heating under confinement or reduction of the desensitized agent.',
      P220: 'Keep away from clothing and other combustible materials.',
      P221: '(Deleted) Take any precaution to avoid mixing with combustibles/...',
      P222: 'Do not allow contact with air.',
      P223: 'Do not allow contact with water.',
      P230: 'Keep wetted with ...',
      P231: 'Handle and store contents under inert gas/...',
      P232: 'Protect from moisture.',
      P233: 'Keep container tightly closed.',
      P234: 'Keep only in original container.',
      P235: 'Keep cool.',
      P236: 'Keep only in original packaging; Division .. in the transport configuration.',
      P240: 'Ground/bond container and receiving equipment.',
      P241: 'Use explosion-proof [electrical/ventilating/lighting/.../] equipment.',
      P242: 'Use only non-sparking tools.',
      P243: 'Take precautionary measures against static discharge.',
      P244: 'Keep valves and fittings free from oil and grease.',
      P250: 'Do not subject to grinding/shock/friction/...',
      P251: 'Do not pierce or burn, even after use.',
      P260: 'Do not breathe dust/fume/gas/mist/vapors/spray.',
      P261: 'Avoid breathing dust/fume/gas/mist/vapors/spray.',
      P262: 'Do not get in eyes, on skin, or on clothing.',
      P263: 'Avoid contact during pregnancy/while nursing.',
      P264: 'Wash hands [and ...] thoroughly after handling.',
      P265: 'Do not touch eyes.',
      P270: 'Do not eat, drink or smoke when using this product.',
      P271: 'Use only outdoors or in a well-ventilated area.',
      P272: 'Contaminated work clothing should not be allowed out of the workplace.',
      P273: 'Avoid release to the environment.',
      P280: 'Wear protective gloves/protective clothing/eye protection/face protection/hearing protection/...',
      P281: '(Deleted)Use personal protective equipment as required.',
      P282: 'Wear cold insulating gloves and either face shield or eye protection.',
      P283: 'Wear fire resistant or flame retardant clothing.',
      P284: '[In case of inadequate ventilation] Wear respiratory protection.',
      P285: '(Deleted) In case of inadequate ventilation wear respiratory protection.',
      'P231+P232': 'Handle and store contents under inert gas/... Protect from moisture.',
      'P264+P265': 'Wash hands [and ...] thoroughly after handling. Do not touch eyes.',
      'P235+P410': '(Deleted) Keep cool. Protect from sunlight.',
      P301: 'IF SWALLOWED:',
      P302: 'IF ON SKIN:',
      P303: 'IF ON SKIN (or hair):',
      P304: 'IF INHALED:',
      P305: 'IF IN EYES:',
      P306: 'IF ON CLOTHING:',
      P307: '(Deleted) IF exposed:',
      P308: 'IF exposed or concerned:',
      P309: '(Deleted) IF exposed or if you feel unwell',
      P310: '(Deleted) Immediately call a POISON CENTER or doctor/physician.',
      P311: '(Deleted) Call a POISON CENTER or doctor/...',
      P312: '(Deleted) Call a POISON CENTER or doctor/... if you feel unwell.',
      P313: '(Deleted) Get medical advice/attention.',
      P314: '(Deleted) Get medical advice/attention if you feel unwell.',
      P315: '(Deleted) Get immediate medical advice/attention.',
      P316: 'Get emergency medical help immediately.',
      P317: 'Get emergency medical help.',
      P318: 'if exposed or concerned, get medical advice.',
      P319: 'Get medical help if you feel unwell.',
      P320: 'Specific treatment is urgent (see ... on this label).',
      P321: 'Specific treatment (see ... on this label).',
      P322: '(Deleted) Specific measures (see ...on this label).',
      P330: 'Rinse mouth.',
      P331: 'Do NOT induce vomiting.',
      P332: 'IF SKIN irritation occurs:',
      P333: 'If skin irritation or rash occurs:',
      P334: 'Immerse in cool water [or wrap in wet bandages].',
      P335: 'Brush off loose particles from skin.',
      P336: 'Thaw frosted parts with lukewarm water. Do not rub affected area.',
      P337: 'If eye irritation persists:',
      P338: 'Remove contact lenses, if present and easy to do. Continue rinsing.',
      P340: 'Remove person to fresh air and keep comfortable for breathing.',
      P341: '(Deleted) If breathing is difficult, remove victim to fresh air and keep at rest in a position comfortable for breathing.',
      P342: 'If experiencing respiratory symptoms:',
      P350: '(Deleted) Gently wash with plenty of soap and water.',
      P351: 'Rinse cautiously with water for several minutes.',
      P352: 'Wash with plenty of water/...',
      P353: 'Rinse skin with water [or shower].',
      P354: 'Immediately rinse with water for several minutes.',
      P360: 'Rinse immediately contaminated clothing and skin with plenty of water before removing clothes.',
      P361: 'Take off immediately all contaminated clothing.',
      P362: 'Take off contaminated clothing.',
      P363: 'Wash contaminated clothing before reuse.',
      P364: 'And wash it before reuse.',
      P370: 'In case of fire:',
      P371: 'In case of major fire and large quantities:',
      P372: 'Explosion risk.',
      P373: 'DO NOT fight fire when fire reaches explosives.',
      P374: '(Deleted) Fight fire with normal precautions from a reasonable distance.',
      P375: 'Fight fire remotely due to the risk of explosion.',
      P376: 'Stop leak if safe to do so.',
      P377: 'Leaking gas fire: Do not extinguish, unless leak can be stopped safely.',
      P378: 'Use ... to extinguish.',
      P380: 'Evacuate area.',
      P381: 'In case of leakage, eliminate all ignition sources.',
      P390: 'Absorb spillage to prevent material damage.',
      P391: 'Collect spillage.',
      'P301+P310': '(Deleted) IF SWALLOWED: Immediately call a POISON CENTER/doctor/...',
      'P301+P312': '(Deleted) IF SWALLOWED: call a POISON CENTER/doctor/... IF you feel unwell.',
      'P301+P316': 'IF SWALLOWED: Get emergency medical help immediately.',
      'P301+P317': 'IF SWALLOWED: Get medical help.',
      'P301+P330+P331': 'IF SWALLOWED: Rinse mouth. Do NOT induce vomiting.',
      'P302+P317': 'IF ON SKIN: Get medical help.',
      'P302+P334': 'IF ON SKIN: Immerse in cool water or wrap in wet bandages.',
      'P302+P335+P334': 'Brush off loose particles from skin. Immerse in cool water [or wrap in wet bandages].',
      'P302+P350': '(Deleted) IF ON SKIN: Gently wash with plenty of soap and water.',
      'P302+P352': 'IF ON SKIN: wash with plenty of water/...',
      'P302+P361+P354': 'IF ON SKIN: Take off Immediately all contaminated clothing. Immediately rinse with water for several minutes.',
      'P303+P361+P353': 'IF ON SKIN (or hair): Take off Immediately all contaminated clothing. Rinse SKIN with water [or shower].',
      'P304+P312': '(Deleted) IF INHALED: Call a POISON CENTER/doctor/... if you feel unwell.',
      'P304+P317': 'IF INHALED: Get medical help.',
      'P304+P340': 'IF INHALED: Remove person to fresh air and keep comfortable for breathing.',
      'P304+P341': '(Deleted) IF INHALED: If breathing is difficult, remove victim to fresh air and keep at rest in a position comfortable for breathing.',
      'P305+P351+P338': 'IF IN EYES: Rinse cautiously with water for several minutes. Remove contact lenses if present and easy to do - continue rinsing.',
      'P305+P354+P338': 'IF IN EYES: Immediately rinse with water for several minutes. Remove contact lenses if present and easy to do. Continue rinsing.',
      'P306+P360': 'IF ON CLOTHING: Rinse Immediately contaminated CLOTHING and SKIN with plenty of water before removing clothes.',
      'P308+P316': 'IF exposed or concerned: Get emergency medical help immediately.',
      'P307+P311': '(Deleted) IF exposed: call a POISON CENTER or doctor/physician.',
      'P308+P311': '(Deleted) IF exposed or concerned: Call a POISON CENTER/doctor/...',
      'P308+P313': '(Deleted) IF exposed or concerned: Get medical advice/attention.',
      'P309+P311': '(Deleted) IF exposed or if you feel unwell: call a POISON CENTER or doctor/physician.',
      'P332+P313': '(Deleted) IF SKIN irritation occurs: Get medical advice/attention.',
      'P332+P317': 'If skin irritation occurs: Get medical help.',
      'P333+P317': 'If skin irritation or rash occurs: Get medical help.',
      'P336+P317': 'Immediately thaw frosted parts with lukewarm water. Do not rub affected area. Get medical help.',
      'P337+P317': 'If eye irritation persists: Get medical help.',
      'P342+P316': 'If experiencing respiratory symptoms: Get emergence medical help immediately.',
      'P333+P313': '(Deleted) IF SKIN irritation or rash occurs: Get medical advice/attention.',
      'P335+P334': '(Deleted) Brush off loose particles from skin. Immerse in cool water/wrap in wet bandages.',
      'P337+P313': '(Deleted) IF eye irritation persists: Get medical advice/attention.',
      'P342+P311': '(Deleted) IF experiencing respiratory symptoms: Call a POISON CENTER/doctor/...',
      'P361+P364': 'Take off immediately all contaminated clothing and wash it before reuse.',
      'P362+P364': 'Take off contaminated clothing and wash it before reuse.',
      'P370+P376': 'in case of fire: Stop leak if safe to do so.',
      'P370+P378': 'In case of fire: Use ... to extinguish.',
      'P370+P380': '(Deleted) In case of fire: Evacuate area.',
      'P370+P380+P375': 'In case of fire: Evacuate area. Fight fire remotely due to the risk of explosion.',
      'P371+P380+P375': 'In case of major fire and large quantities: Evacuate area. Fight fire remotely due to the risk of explosion.',
      'P370+P372+P380+P373': 'In case of fire: Explosion risk. Evacuate area. DO NOT fight fire when fire reaches explosives.',
      'P370+P380+P375[+P378]': 'In case of fire: Evacuate area. Fight fire remotely due to the risk of explosion. [Use...to extinguish].]',
      P401: 'Store in accordance with ...',
      P402: 'Store in a dry place.',
      P403: 'Store in a well-ventilated place.',
      P404: 'Store in a closed container.',
      P405: 'Store locked up.',
      P406: 'Store in corrosive resistant/... container with a resistant inner liner.',
      P407: 'Maintain air gap between stacks or pallets.',
      P410: 'Protect from sunlight.',
      P411: 'Store at temperatures not exceeding ... °C/...°F.',
      P412: 'Do not expose to temperatures exceeding 50 °C/ 122 °F.',
      P413: 'Store bulk masses greater than ... kg/...lbs at temperatures not exceeding ... °C/...°F.',
      P420: 'Store separately.',
      P422: '(Deleted) Store contents under ...',
      'P402+P404': 'Store in a dry place. Store in a closed container.',
      'P403+P233': 'Store in a well-ventilated place. Keep container tightly closed.',
      'P403+P235': 'Store in a well-ventilated place. Keep cool.',
      'P410+P403': 'Protect from sunlight. Store in a well-ventilated place.',
      'P410+P412': 'Protect from sunlight. Do not expose to temperatures exceeding 50 °C/122°F.',
      'P411+P235': '(Deleted) Store at temperatures not exceeding ... °C/...°F. Keep cool.',
      P501: 'Dispose of contents/container to ...',
      P502: 'Refer to manufacturer or supplier for information on recovery or recycling',
      P503: 'Refer to manufacturer/supplier... for information on disposal/recovery/recycling.'
    };

    /**
     * Extracts GHS information (H/P-Statements and pictograms)
     * @param data The data to extract from
     * @return The extracted data
     */
    function getGHS(data) {
      const references = getReferences(data);
      let pictograms;
      try {
        pictograms = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="Pictogram(s)")]').map(entry => ({
          reference: references[entry.ReferenceNumber],
          data: jp.query(entry, '$.Value.StringWithMarkup[*].Markup[*]').map(entry => {
            const code = entry.URL.match(/GHS\d+/)[0];
            return {
              code,
              description: ghsPictogramText[code]
            };
          })
        }));
      } catch (error) {
        pictograms = [];
      }
      let hStatements;
      try {
        hStatements = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="GHS Hazard Statements")]').map(entry => ({
          reference: references[entry.ReferenceNumber],
          data: jp.query(entry, '$.Value.StringWithMarkup[*]').map(entry => entry.String.match(/H\d+/)[0]).map(hCode => ({
            code: hCode,
            description: hazardStatements[hCode]
          }))
        }));
      } catch (error) {
        hStatements = [];
      }
      //ToDo(kjappelbaum): investigate in more detail why they do not have the full P statements
      //For P statements the full sentence (with conditions) is more important than just the number
      let pStatements;
      try {
        pStatements = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="Precautionary Statement Codes")]').map(entry => ({
          reference: references[entry.ReferenceNumber],
          data: jp.query(entry, '$.Value.StringWithMarkup[*]').map(entry => entry.String.match(
          // eslint-disable-next-line no-lookahead-lookbehind-regexp/no-lookahead-lookbehind-regexp
          /(?<oneP>(?<!\+)P\d\d\d(?!\+))|(?<twoP>P\d\d\d\+P\d\d\d(?!\+))|(?<threeP>P\d\d\d\+P\d\d\d\+P\d\d\d(?!\+))/gm))[0].map(pCode => ({
            code: pCode,
            description: precautionaryStatements[pCode]
          }))
        }));
      } catch (error) {
        pStatements = [];
      }
      return {
        pictograms,
        hStatements,
        pStatements
      };
    }
    /**
     * We combine all the results and keep
     */
    function getGHSSummary(data, options = {}) {
      const {
        sourceName
      } = options;
      const result = getGHS(data);
      if (sourceName) {
        result.pictograms = result.pictograms.filter(entry => entry.reference.sourceName.match(sourceName));
        result.hStatements = result.hStatements.filter(entry => entry.reference.sourceName.match(sourceName));
        result.pStatements = result.pStatements.filter(entry => entry.reference.sourceName.match(sourceName));
      }
      return {
        pictograms: getUnique(result.pictograms),
        hStatements: getUnique(result.hStatements),
        pStatements: getUnique(result.pStatements)
      };
    }
    function getUnique(entries) {
      const uniqueMap = {};
      entries.map(entry => entry.data.forEach(line => {
        uniqueMap[line.code] = line;
      }));
      return Object.keys(uniqueMap).map(code => uniqueMap[code]).sort((a, b) => a.code < b.code ? -1 : 1);
    }

    class CompoundData {
      constructor(data) {
        this.data = data;
      }
      getReferences() {
        return getReferences(this.data);
      }
      getExperimentalData(options) {
        return getExperimentalData(this.data, options);
      }
      getIdentifiers() {
        return getIdentifiers(this.data);
      }
      get computed() {
        return getComputedData(this.data);
      }
      /**
       * Property containing a summary of GHS information
       */
      get ghs() {
        return getGHSSummary(this.data);
      }
      getGHSSummary(options) {
        return getGHSSummary(this.data, options);
      }
      /**
       * Canonical SMILES
       */
      getSMILES() {
        return getSMILES(this.data);
      }
      /**
       * Return molecular formula in Hill notation
       */
      getFormula() {
        return getFormula(this.data);
      }
      /**
       * Return the International Chemical Identifier (InChI) computed from chemical structure using the International Union of Pure and Applied Chemistry (IUPAC) standard
       */
      getInChI() {
        return getInChI(this.data);
      }
      /**
       * International Chemical Identifier hash (InChIKey) computed from chemical structure using the International Union of Pure and Applied Chemistry (IUPAC) standard.
       */
      getInChIKey() {
        return getInChIKey(this.data);
      }
      /**
       * Detailed information about GHS information
       */
      getGHS() {
        return getGHS(this.data);
      }
      getMeltingPoint(options) {
        return getMeltingPoint(this.data, options);
      }
      getBoilingPoint(options) {
        return getBoilingPoint(this.data, options);
      }
      getDensity(options) {
        return getDensity(this.data, options);
      }
      getVaporPressure(options) {
        return getVaporPressure(this.data, options);
      }
      getSolubility() {
        return getSolubility(this.data);
      }
      getFlashPoint(options) {
        return getFlashPoint(this.data, options);
      }
      toJSON() {
        const methods = Object.entries(Object.getOwnPropertyDescriptors(CompoundData.prototype)).filter(([, descriptor]) => typeof descriptor.get === 'function').map(([key]) => key);
        const result = {};
        for (const method of methods) {
          result[method] = this[method];
        }
        return result;
      }
    }

    async function compoundDataFromCID(cid, options = {}) {
      const {
        cache
      } = options;
      let compoundData;
      if (cache?.(cid)) {
        compoundData = cache(cid);
      } else {
        const dataResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug_view/data/compound/${cid}/JSON`);
        compoundData = (await dataResponse.json()).Record;
        if (cache) {
          cache(cid, compoundData);
        }
      }
      return new CompoundData(compoundData);
    }

    function checkCompoundsResult(compounds) {
      if (!Array.isArray(compounds) || compounds.length === 0) {
        throw new Error('No compound found');
      }
      if (compounds.length !== 1) {
        throw new Error('More than one compound found');
      }
      if (!('id' in compounds[0].id)) {
        throw new Error('No compound found');
      }
    }

    async function compoundFromInchiKey(inchiKey, options = {}) {
      const {
        cache
      } = options;
      let compounds;
      if (cache?.(inchiKey)) {
        compounds = cache(inchiKey);
      } else {
        const encodedInchiKey = encodeURIComponent(inchiKey);
        const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/inchikey/${encodedInchiKey}/json`);
        compounds = (await moleculeResponse.json()).PC_Compounds;
        if (cache) {
          cache(inchiKey, compounds);
        }
      }
      checkCompoundsResult(compounds);
      return new Compound(compounds[0], {
        cache
      });
    }

    async function compoundFromName(smiles, options = {}) {
      const {
        cache
      } = options;
      let compounds;
      if (cache?.(smiles)) {
        compounds = cache(smiles);
      } else {
        const encodedName = encodeURIComponent(smiles);
        const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/name/${encodedName}/json`);
        compounds = (await moleculeResponse.json()).PC_Compounds;
        if (cache) {
          cache(smiles, compounds);
        }
      }
      checkCompoundsResult(compounds);
      return new Compound(compounds[0], {
        cache
      });
    }

    async function compoundFromSmiles(smiles, options = {}) {
      const {
        cache
      } = options;
      let compounds;
      if (cache?.(smiles)) {
        compounds = cache(smiles);
      } else {
        const encodedSmiles = encodeURIComponent(smiles);
        const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/smiles/${encodedSmiles}/json`);
        compounds = (await moleculeResponse.json()).PC_Compounds;
        if (cache) {
          cache(smiles, compounds);
        }
      }
      checkCompoundsResult(compounds);
      return new Compound(compounds[0], {
        cache
      });
    }

    class Compound {
      constructor(data, options = {}) {
        this.data = data;
        this.cache = options.cache;
      }
      getCID() {
        return this.data?.id?.id?.cid;
      }
      getData() {
        const cid = this.getCID();
        const cd = compoundDataFromCID(cid, {
          cache: this.cache
        });
        return cd;
      }
      toJSON() {
        const methods = Object.entries(Object.getOwnPropertyDescriptors(Compound.prototype)).filter(([, descriptor]) => typeof descriptor.get === 'function').map(([key]) => key);
        const result = {};
        for (const method of methods) {
          result[method] = this[method];
        }
        return result;
      }
    }
    Compound.fromSmiles = compoundFromSmiles;
    Compound.fromName = compoundFromName;
    Compound.fromInchiKey = compoundFromInchiKey;

    /**
     * We check that the hazards are present in the reagents
     * @param {object} reagent
     * @param options
     */
    async function ensureHazardAndPhysical(reagent, options = {}) {
      const {
        logger
      } = options;
      if (reagent.hazards && reagent.hStatements) return;
      if (reagent.hazards) {
        if (typeof reagent.hazards === 'string') {
          reagent.hazards = reagent.hazards.split(',').map(h => h.trim());
        }
        reagent.hStatements = reagent.hazards.map(h => getHStatementsObject()[h]);
        return;
      }
      try {
        const compound = await Compound.fromSmiles(reagent.smiles);
        const data = await compound.getData();
        const summary = data.getGHSSummary({
          sourceName: 'European'
        });
        reagent.hStatements = summary.hStatements;
        reagent.hazards = reagent.hStatements.map(h => h.code);
      } catch (error) {
        logger.error(`Could not get hazards for ${reagent.smiles}: ${error.toString()}`);
        reagent.hStatements = [];
        reagent.hazards = [];
      }
    }

    var openchemlibCore = {};

    (function(exports){(function(root){function getExports($wnd){var $doc=$wnd.document;var $gwt={};var navigator={userAgent:'webkit'};function noop(){}var __gwtModuleFunction=noop;__gwtModuleFunction.__moduleStartupDone=noop;var osb='object',psb='anonymous',qsb='fnStack',rsb='\n',ssb={4:1,10:1,5:1,8:1},tsb='Unknown',usb='boolean',vsb='number',wsb='string',xsb=2147483647,ysb='For input string: "',zsb='null',Asb='__noinit__',Bsb={4:1,5:1,8:1},Csb={4:1,15:1,26:1,23:1},Dsb=' (copy)',Esb={4:1,10:1,32:1,5:1,17:1,8:1,30:1},Fsb=65536,Gsb=65535,Hsb=10000,Isb='fromIndex: ',Jsb=' > toIndex: ',Ksb=', toIndex: ',Lsb=', length: ',Msb='Index: ',Nsb=', Size: ',Osb={12:1,4:1,5:1},Psb={4:1},Qsb={7:1,4:1,5:1},Rsb=16777215,Ssb=65280,Tsb=0.30000001192092896,Usb=545460846592,Vsb=4294967296,Wsb=8589934592,Xsb=17179869184,Ysb=34359738368,Zsb=68719476736,$sb=137438953472,_sb=274877906944,atb='<line x1="',btb='" ',ctb='y1="',dtb='x2="',etb='y2="',ftb='stroke="',gtb='stroke-width="',htb='" />',itb={11:1,4:1,5:1},jtb=536870912,ktb=2.617993878,ltb=3.665191429,mtb=6.283185307179586,ntb=3.141592653589793,otb=1.5707963267948966,ptb=16384,qtb=4096,rtb={l:0,m:0,h:1},stb={l:0,m:0,h:2},ttb={l:0,m:0,h:4},utb=1920,vtb=1024,wtb=234881024,xtb=167772160,ytb=100663296,ztb=201326592,Atb=114688,Btb=4063232,Ctb=3538944,Dtb=3014656,Etb=2097152,Ftb=393216,Gtb=917504,Htb=1966080,Itb=17042430230528,Jtb=16492674416640,Ktb=15942918602752,Ltb=14843406974976,Mtb=12644383719424,Ntb=8796093022208,Otb=549755813888,Ptb=1649267441664,Qtb=3848290697216,Rtb=8246337208320,Stb=29360128,Ttb=268435456,Utb=8192,Vtb=2048,Wtb=-1.5707963267948966,Xtb=130560,Ytb=6291456,Ztb=1.0471975511965976,$tb=0.5235987755982988,_tb=786432,aub=524288,bub=262144,cub=-16777216,dub=16711680,eub=-32768,fub='<DrawingObject type="',gub='><\/DrawingObject>',hub={4:1,5:1,13:1},iub={4:1,5:1},jub='unsupported atomicNo:',kub=2.8415926535897933,lub={25:1,4:1,10:1,5:1,8:1},mub={4:1,5:1,17:1,8:1},nub={9:1,4:1,5:1},oub=131072,pub={l:6,m:0,h:4},qub=1048576,rub=3221225472,tub={l:0,m:0,h:3},uub=8388608,vub=-65536,wub={33:1,42:1,36:1},xub={4:1,10:1,5:1,17:1,8:1},yub={33:1,42:1,77:1,36:1,72:1},zub={4:1,5:1,33:1,42:1,77:1,18:1,36:1,72:1,108:1},Aub='??',Bub=2013265920,Cub=32505856,Dub=31744,Eub=1.7976931348623157E308,Fub={113:1,4:1,10:1,5:1,17:1,8:1},Gub=33554432,Hub=67108864,Iub=32768,Jub=-131073,Kub=-33554433,Lub=-33292289,Mub=-3.141592653589793,Nub=4194303,Oub=3072,Pub={40:1,4:1,5:1},Qub=0.7853981633974483,Rub=3.061592653589793,Sub=1073741824,Tub=541165879296,Uub={14:1,4:1,5:1,8:1},Vub=281.17,Wub=289.19,Xub=294.21,Yub='0.0000',Zub='M  END\n',$ub='ATOMS',_ub=-2147483648,avb='M  END',bvb='$$$$',cvb='M  V30 ',dvb='M  V30 COUNTS ',evb=')\n',fvb='M  V30 MDLV30/STEREL',gvb='M  V30 MDLV30/STERAC',hvb=3.4028234663852886E38,ivb={l:2,m:0,h:4},jvb=4194304,kvb={l:4179966,m:4063288,h:7},lvb={l:4194302,m:4193343,h:7},mvb={126:1,111:1,112:1,4:1,10:1,5:1,8:1},nvb='class="event" ',ovb='opacity="0" />',pvb="' position:",qvb="'. Position:",rvb=1099511627776,svb=2199023255552,tvb=4398046511104,uvb="', position:",vvb='Assignment of aromatic double bonds failed',wvb=2147483646,xvb='Members of ESR groups must only be stereo centers with known configuration.',yvb='Over- or under-specified stereo feature or more than one racemic type bond',zvb='Ambiguous configuration at stereo center because of 2 parallel bonds',Avb={101:1,85:1,27:1,4:1},Bvb=100000,Cvb=3.1415927410125732,Dvb=1.5707963705062866,Evb=0.029999999329447746,Fvb=6.2831854820251465,Gvb=1.9106119871139526,Hvb=2.2639999389648438,Ivb=2.16759991645813,Jvb=2.0662999153137207,Kvb=2.3561999797821045,Lvb=2.2844998836517334,Mvb=1.9106120066311025,Nvb=2.094395160675049,Ovb=0.6000000238418579,Pvb=1.7999999523162842,Qvb=0.05000000074505806,Rvb={4:1,10:1,5:1,8:1,31:1},Svb=0.2617993950843811,Tvb=2.0943951023931953,Uvb=1.2000000476837158,Vvb=1.4299999475479126,Wvb=1.659999966621399,Xvb=1.4600000381469727,Yvb=1.8200000524520874,Zvb=2.4600000381469727,$vb=2.4200000762939453,_vb=2.450000047683716,awb=2.440000057220459,bwb=2.4000000953674316,cwb=2.319999933242798,dwb=2.2899999618530273,ewb=1.8600000143051147,fwb=2.5299999713897705,gwb=2.490000009536743,hwb=2.4700000286102295,iwb=2.880000114440918,jwb=2.9200000762939453,kwb=2.869999885559082,lwb=2.8299999237060547,mwb=2.7899999618530273,nwb=2.809999942779541,owb=2.799999952316284,pwb=3.049999952316284,qwb=1.3300000429153442,rwb=1.600000023841858,swb=1.3899999856948853,twb=1.159999966621399,uwb=1.1100000143051147,vwb=1.0700000524520874,wwb=1.9600000381469727,xwb=1.7100000381469727,ywb=1.3600000143051147,zwb=1.340000033378601,Awb=1.2200000286102295,Bwb=1.2400000095367432,Cwb=1.2100000381469727,Dwb=1.850000023841858,Ewb=1.6299999952316284,Fwb=1.4700000286102295,Gwb=1.2799999713897705,Hwb=1.440000057220459,Iwb=1.399999976158142,Jwb=1.7599999904632568,Kwb=1.7300000190734863,Lwb=1.7200000286102295,Mwb=1.6799999475479126,Nwb=1.690000057220459,Owb=1.6699999570846558,Pwb=1.649999976158142,Qwb=1.7000000476837158,Rwb=1.6200000047683716,Swb=1.3700000047683716,Twb=1.2899999618530273,Uwb=1.4500000476837158,Vwb=1.5700000524520874,Wwb=-0.5235987755982988,Xwb=-1.0471975511965976,Ywb=-0.7853981633974483,Zwb=0.017453292519943295,$wb=0.17453292519943295,_wb=1.0E-4,axb={123:1},bxb=57.29577951308232,cxb=0.021922173386725217,dxb=1.0E-8,exb=2.5120960256267386,fxb=1.0E-5,gxb='MMFF94s+',hxb='nonbonded cutoff',ixb='dielectric constant',jxb='dielectric model',kxb='angle bend',lxb='bond stretch',mxb='electrostatic',nxb='out of plane',oxb='stretch bend',pxb='torsion angle',qxb='van der waals',rxb=0.07000000000000006,sxb=0.1200000000000001,txb=-7.840000000000001,uxb='OCL_RXN_V1.0:',vxb="'$MOL' tag not found",wxb="'M  END' not found",xxb='\nActelion Java MolfileCreator 2.0\n\n  0  0  0  0  0  0              0 V3000\n',yxb='M  V30 END CTAB',zxb=2475109711874,Axb=2475109711875,Bxb=2887426572290,Cxb=2887426572291,Dxb=2887627898882,Exb=2887627898883,Fxb=2887627997186,Gxb=3437182386178,Hxb=3437182386179,Ixb=3437383712770,Jxb=3437383811074,Kxb=3437652148226,Lxb=3437652377602,Mxb=4674132967426,Nxb=4674334294018,Oxb=4674334392322,Pxb=4674602729474,Qxb=4674602958850,Rxb=5086449827842,Sxb=5086449827848,Txb=5086651154434,Uxb=5086651252738,Vxb=5086651252743,Wxb=5086919589890,Xxb=5086919688194,Yxb=5636406968322,Zxb=5636407066626,$xb=5636675502082,_xb={l:589826,m:590112,h:4},ayb={l:589827,m:590112,h:4},byb={l:589826,m:590160,h:4},cyb={l:589827,m:590160,h:4},dyb={l:688130,m:590160,h:4},eyb={l:589826,m:590224,h:4},fyb={l:589827,m:590224,h:4},gyb={l:819203,m:590224,h:4},hyb={l:589826,m:590368,h:4},iyb={l:688130,m:590368,h:4},jyb={l:819202,m:590368,h:4},kyb={l:819208,m:590368,h:4},lyb={l:589826,m:590416,h:4},myb={l:589832,m:590416,h:4},nyb={l:688130,m:590416,h:4},oyb={l:819202,m:590416,h:4},pyb={l:589826,m:598304,h:4},qyb={l:589827,m:598304,h:4},ryb={l:688130,m:598304,h:4},syb={l:1114114,m:598304,h:4},tyb={l:1212418,m:598304,h:4},uyb={l:589826,m:598308,h:4},vyb={l:589826,m:688416,h:4},wyb={l:589826,m:688464,h:4},xyb={l:589826,m:688720,h:4},yyb={l:688130,m:688720,h:4},zyb={l:589826,m:590112,h:288},Ayb={l:589826,m:590112,h:336},Byb={l:589826,m:688416,h:336},Cyb={l:589826,m:688464,h:336},Dyb={l:688130,m:688464,h:336},Eyb={l:589826,m:590112,h:400},Fyb={l:589826,m:688416,h:400},Gyb={l:589826,m:688464,h:400},Hyb={l:688130,m:688464,h:400},Iyb={l:589826,m:819488,h:400},Jyb={l:589826,m:819536,h:400},Kyb={l:589826,m:819600,h:400},Lyb={l:688130,m:819600,h:400},Myb={l:819202,m:819600,h:400},Nyb={l:589831,m:688416,h:592},Oyb={l:589831,m:688464,h:592},Pyb={l:688135,m:688464,h:592},Qyb={l:589831,m:819536,h:592},Ryb={l:688135,m:819536,h:592},Syb={l:688135,m:819600,h:592},Tyb={l:589832,m:1212704,h:592},Uyb={l:589832,m:1212752,h:592},Vyb={l:688136,m:1212752,h:592},Wyb={l:589831,m:688464,h:656},Xyb={l:589826,m:590112,h:8480},Yyb={l:589826,m:688416,h:8480},Zyb={l:589826,m:688464,h:8480},$yb={l:688130,m:688464,h:8480},_yb={l:819202,m:819600,h:8480},azb={l:688135,m:1212752,h:8480},bzb={l:589832,m:1213008,h:8480},czb={l:688136,m:1213008,h:8480},dzb={l:589826,m:590112,h:8484},ezb={l:589826,m:590160,h:8484},fzb={l:688130,m:598304,h:8484},gzb=0.6262000203132629,hzb=-1.3825000524520874,izb=0.33169999718666077,jzb=-1.4915000200271606,kzb=0.3540000021457672,lzb=0.38179999589920044,mzb=-0.6019999980926514,nzb=-0.7379999756813049,ozb='Atom-types are 64-bit numbers describing atoms and their near surrounding.',pzb='Recognized atom types and their contributions are:',qzb='Druglikeness predictor not properly initialized.',rzb=3.009999990463257,szb=-0.17000000178813934,tzb=-0.1809999942779541,uzb=-0.2029999941587448,vzb=2147483648,wzb='MMFF94',xzb='MMFF94s',yzb='117\nint,int,int,int,float\n0,2,0,0,0.020\n1,2,1,2,0.030\n1,2,2,2,0.027\n1,2,2,3,0.026\n1,2,2,5,0.013\n1,2,2,37,0.032\n2,2,2,5,0.013\n2,2,3,5,0.012\n2,2,5,5,0.006\n2,2,5,6,0.027\n2,2,5,37,0.017\n2,2,5,40,0.012\n2,2,5,41,0.008\n0,3,0,0,0.130\n1,3,1,7,0.146\n1,3,2,7,0.138\n1,3,3,7,0.134\n1,3,5,7,0.122\n1,3,6,7,0.141\n1,3,7,10,0.129\n1,3,7,37,0.138\n2,3,5,7,0.113\n2,3,5,9,0.081\n2,3,6,7,0.127\n2,3,7,10,0.116\n3,3,5,7,0.113\n3,3,6,7,0.127\n5,3,5,7,0.103\n5,3,5,9,0.074\n5,3,5,54,0.078\n5,3,6,7,0.119\n5,3,7,10,0.102\n5,3,9,40,0.067\n6,3,7,37,0.127\n7,3,10,10,0.113\n7,3,20,20,0.151\n9,3,40,40,0.057\n0,8,0,0,0.000\n0,10,0,0,0.015\n1,10,1,3,0.015\n1,10,3,6,0.015\n1,10,3,28,0.015\n3,10,3,28,0.015\n3,10,28,28,0.015\n0,17,0,0,0.000\n0,26,0,0,0.000\n0,30,0,0,0.010\n5,30,20,30,0.008\n0,37,0,0,0.035\n1,37,37,37,0.040\n2,37,37,37,0.031\n3,37,37,37,0.027\n5,37,37,37,0.015\n5,37,37,38,0.046\n5,37,37,63,0.008\n5,37,37,64,0.012\n5,37,37,69,0.016\n5,37,38,38,0.084\n6,37,37,37,0.048\n15,37,37,37,0.025\n37,37,37,40,0.046\n0,39,0,0,0.020\n1,39,63,63,0.012\n23,39,63,63,-0.014\n23,39,63,65,0.021\n23,39,65,65,0.062\n0,40,0,0,0.030\n1,40,28,37,0.030\n2,40,28,28,0.030\n3,40,28,28,0.030\n28,40,28,37,0.030\n0,41,0,0,0.180\n1,41,32,32,0.178\n2,41,32,32,0.161\n5,41,32,32,0.158\n0,43,0,0,0.000\n0,45,0,0,0.150\n0,49,0,0,0.000\n50,49,50,50,0.000\n0,54,0,0,0.020\n1,54,3,36,0.016\n3,54,36,36,0.018\n0,55,0,0,0.020\n1,55,36,57,0.020\n36,55,36,57,0.020\n0,56,0,0,0.020\n1,56,36,57,0.020\n36,56,36,57,0.020\n0,57,0,0,0.080\n5,57,55,55,0.038\n56,57,56,56,0.158\n0,58,0,0,0.025\n0,63,0,0,0.050\n5,63,39,64,0.019\n5,63,39,66,0.068\n5,63,44,64,0.014\n5,63,44,66,0.055\n5,63,59,64,0.033\n5,63,59,66,0.085\n37,63,39,64,0.010\n0,64,0,0,0.040\n5,64,63,64,0.006\n5,64,63,66,0.043\n5,64,64,65,0.052\n5,64,65,66,0.094\n37,64,63,64,-0.01\n0,67,0,0,0.070\n0,69,0,0,0.070\n32,69,37,37,0.067\n0,73,0,0,0.000\n0,78,0,0,0.045\n5,78,78,81,0.046\n0,80,0,0,0.080\n5,80,81,81,0.057\n0,81,0,0,0.025\n36,81,78,80,0.016\n0,82,0,0,0.000\n',zzb={140:1,4:1,5:1,8:1},Azb='undefined',Bzb=0.08726646502812703,Czb=0.003921568859368563,Dzb='Too many percent/per mille characters in pattern "',Ezb=1048575,Fzb=17592186044416,Gzb=1000000000,Hzb=-17592186044416,Izb='CSS1Compat',Jzb=5.56,Kzb=11.12,Lzb=13.34,Mzb=14.44,Nzb=0.3010299956639812,Ozb='BigInteger divide by zero',Pzb=4294967295,Qzb={l:0,m:0,h:524288},Rzb={28:1,96:1},Szb={94:1},Tzb={33:1,42:1,36:1,74:1},Uzb=16777619,Vzb={33:1,36:1,74:1},Wzb='delete',Xzb=15525485,Yzb=5.960464477539063E-8,Zzb=16777216,$zb={4:1,28:1,49:1,69:1},_zb='Invalid UTF8 sequence',aAb=0.20000000298023224,bAb=' distance:',cAb='gGP`@dfyjidNcGI[WQCP`<',dAb={45:1};var Drb={};var Erb={};var Frb={};var Grb={};var Hrb={};var Irb={};var Jrb={};var Krb={};var Lrb={};var Mrb={};var Nrb={};var Orb={};var Prb={};var Qrb={};var Rrb={};var Srb={};var Trb={};var Urb={};var Vrb={};var Wrb={};var Xrb={};var Yrb={};var Zrb={};var $rb={};var _rb={};var asb={};var bsb={};var csb={};var dsb={};var esb={};var fsb={};var gsb={};var hsb={};var isb={};var jsb={};var ksb={};var lsb={};var msb={};var nsb={};var _;var W3;var Q3;Drb.e3=function e3(){};function V3(a,b){typeof window===osb&&typeof window['$gwt']===osb&&(window['$gwt'][a]=b);}function U3(b,c,d,e){Drb.T3();var f=Q3;function g(){for(var a=0;a<f.length;a++){f[a]();}}if(b){try{Crb(g)();}catch(a){b(c,a);}}else {Crb(g)();}}Drb.T3=function T3(){Q3==null&&(Q3=[]);};function S3(){Drb.T3();var a=Q3;for(var b=0;b<arguments.length;b++){a.push(arguments[b]);}}Drb.R3=function R3(){};function f4(){}function e4(a){if(Array.isArray(a)&&a.Dc===f4){return Erb.Rc(a);}return a.toString();}function d4(a,b){var c=$wnd;if(a===''){return c;}var d=a.split('.');!(d[0]in c)&&c.execScript&&c.execScript('var '+d[0]);if(b){var e=b.prototype.Bc;e.e=b;}for(var f;d.length&&(f=d.shift());){c=c[f]=c[f]||!d.length&&b||{};}return c;}Drb.c4=function c4(a){function b(){}b.prototype=a||{};return new b();};Drb.b4=function b4(a){return a instanceof Array?a[0]:null;};function a4(a,b,c){var d=function(){return a.apply(d,arguments);};b.apply(d,c);return d;}function _3(){}function $3(a,b){for(var c in b){b[c]['configurable']=true;}Object.defineProperties(a,b);}function Z3(a,b,c){var d=W3;var e=d[a];var f=Drb.b4(e);if(e&&!f){_=e;}else {_=Drb.Y3(b);_.Cc=c;!b&&(_.Dc=f4);d[a]=_;}for(var g=3;g<arguments.length;++g){arguments[g].prototype=_;}f&&(_.Bc=f);}Drb.Y3=function Y3(a){var b=a&&a.prototype;!b&&(b=W3[a]);return Drb.c4(b);};Drb.X3=function X3(){};$wnd.goog=$wnd.goog||{};$wnd.goog.global=$wnd.goog.global||$wnd;W3={};Erb.Gc=function Gc(a,b){return HY(a)===HY(b);};Erb.Hc=function Hc(a){return a.Bc;};Erb.Ic=function Ic(a){return Hrb.$mb(a);};function Jc(){}Erb.Lc=function Lc(a,b){return EY(a)?Erb.k7(a,b):BY(a)?Erb._5(a,b):AY(a)?Erb.U4(a,b):yY(a)?a.jb(b):vX(a)?Erb.Gc(a,b):Frb.FU(a,b);};Erb.Nc=function Nc(a){return EY(a)?Erb.m7():BY(a)?Erb.a6():AY(a)?Erb.V4():yY(a)?a.Bc:vX(a)?Erb.Hc(a):Frb.GU(a);};Erb.Pc=function Pc(a){return EY(a)?Erb.n7(a):BY(a)?Erb.b6(a):AY(a)?Erb.W4(a):yY(a)?a.lb():vX(a)?Erb.Ic(a):Frb.HU(a);};Erb.Rc=function Rc(a){var b;return Erb.v5(Erb.Nc(a))+'@'+(b=Hrb.rnb(Hrb.qnb(Erb.Pc(a))),b.toString(16));};Z3(1,null,{},Jc);_.jb=function Kc(a){return Erb.Gc(this,a);};_.kb=function Mc(){return Erb.Hc(this);};_.lb=function Oc(){return Erb.Ic(this);};_.mb=function Qc(){var a;return Erb.v5(Erb.Nc(this))+'@'+(a=Hrb.rnb(Hrb.qnb(Erb.Pc(this))),a.toString(16));};_.equals=function(a){return this.jb(a);};_.hashCode=function(){return this.lb();};_.toString=function(){return this.mb();};Frb.FU=function FU(a,b){return Frb.LU(a)?Frb.IU(a,b):HY(a)===HY(b);};Frb.GU=function GU(a){return a.Bc||Array.isArray(a)&&nX(Drb.H_,1)||Drb.H_;};Frb.HU=function HU(a){return Frb.MU(a)?Frb.JU(a):Hrb.$mb(a);};Frb.IU=function IU(a,b){return a.equals(b);};Frb.JU=function JU(a){return a.hashCode();};Frb.KU=function KU(){return [];};Frb.LU=function LU(a){return !!a&&!!a.equals;};Frb.MU=function MU(a){return !!a&&!!a.hashCode;};Frb.NU=function NU(a){return a.toString?a.toString():'[JavaScriptObject]';};Grb.oW=function oW(){Grb.oW=_3;var a,b;b=!Grb.xW();a=new Grb.GW();Grb.nW=b?new Grb.yW():a;};Grb.pW=function pW(a){Grb.oW();Grb.nW.Xb(a);};Grb.qW=function qW(a){Grb.oW();var b;b=Grb.nW.Yb(a);return Grb.rW(b);};Grb.rW=function rW(a){var b,c,d,e;b='pW';c='Wz';e=$wnd.Math.min(a.length,5);for(d=e-1;d>=0;d--){if(Erb.k7(a[d].d,b)||Erb.k7(a[d].d,c)){a.length>=d+1&&Hrb.rnb(a).splice(0,d+1);break;}}return a;};Grb.sW=function sW(a){var b=/function(?:\s+([\w$]+))?\s*\(/;var c=b.exec(a);return c&&c[1]||psb;};Grb.tW=function tW(a){Grb.oW();return a&&a[qsb]?a[qsb]:[];};Grb.uW=function uW(a){Grb.oW();return a.name||(a.name=Grb.sW(a.toString()));};Grb.vW=function vW(a){Grb.oW();return parseInt(a)||-1;};Grb.wW=function wW(a){Grb.oW();var b=a.backingJsObject;if(b&&b.stack){var c=b.stack;var d=b+rsb;c.substring(0,d.length)==d&&(c=c.substring(d.length));return c.split(rsb);}return [];};Grb.xW=function xW(){if(Error.stackTraceLimit>0){$wnd.Error.stackTraceLimit=Error.stackTraceLimit=64;return true;}return 'stack'in new Error();};Z3(376,1,{});Grb.yW=function yW(){};Z3(255,376,{},Grb.yW);_.Xb=function zW(a){var b={};var c=[];a[qsb]=c;var d=arguments.callee.caller;while(d){var e=Grb.uW(d);c.push(e);var f=':'+e;var g=b[f];if(g){var h,i;for(h=0,i=g.length;h<i;h++){if(g[h]===d){return;}}}(g||(b[f]=[])).push(d);d=d.caller;}};_.Yb=function AW(a){var b,c,d,e;d=Grb.tW(a);c=Frb.QV(d);e=tX(Drb.x0,ssb,75,c,0,1);for(b=0;b<c;b++){e[b]=new Erb.$6(Frb.PV(d,b),null,-1);}return e;};Grb.BW=function BW(a,b){var c,d,e,f,g,h,i,j,k;j='';if(Hrb.rnb(b).length==0){return a.Zb(tsb,psb,-1,-1);}k=Erb.y7(b);Erb.k7(Hrb.rnb(k).substr(0,3),'at ')&&(k=(Hrb.mnb(3,Hrb.rnb(k).length+1),Hrb.rnb(k).substr(3)));k=Grb.CW(k);g=Hrb.rnb(k).indexOf('(');if(g==-1){g=Hrb.rnb(k).indexOf('@');if(g==-1){j=k;k='';}else {j=Erb.y7((Hrb.mnb(g+1,Hrb.rnb(k).length+1),Hrb.rnb(k).substr(g+1)));k=Erb.y7((Hrb.lnb(0,g,Hrb.rnb(k).length),Hrb.rnb(k).substr(0,g)));}}else {c=Hrb.rnb(k).indexOf(')',g);j=(Hrb.lnb(g+1,c,Hrb.rnb(k).length),Hrb.rnb(k).substr(g+1,c-(g+1)));k=Erb.y7((Hrb.lnb(0,g,Hrb.rnb(k).length),Hrb.rnb(k).substr(0,g)));}g=Erb.o7(k,A7(46));g!=-1&&(k=(Hrb.mnb(g+1,Hrb.rnb(k).length+1),Hrb.rnb(k).substr(g+1)));(Hrb.rnb(k).length==0||Erb.k7(k,'Anonymous function'))&&(k=psb);h=Erb.r7(j,A7(58));e=Erb.s7(j,A7(58),h-1);i=-1;d=-1;f=tsb;if(h!=-1&&e!=-1){f=(Hrb.lnb(0,e,Hrb.rnb(j).length),Hrb.rnb(j).substr(0,e));i=Grb.vW((Hrb.lnb(e+1,h,Hrb.rnb(j).length),Hrb.rnb(j).substr(e+1,h-(e+1))));d=Grb.vW((Hrb.mnb(h+1,Hrb.rnb(j).length+1),Hrb.rnb(j).substr(h+1)));}return a.Zb(f,k,i,d);};Grb.CW=function CW(a){return a.replace(/\[.*?\]/g,'');};Z3(377,376,{});_.Xb=function DW(a){};_.Zb=function EW(a,b,c,d){return new Erb.$6(b,a+'@'+d,c<0?-1:c);};_.Yb=function FW(a){var b,c,d,e,f,g;e=Grb.wW(a);f=tX(Drb.x0,ssb,75,0,0,1);b=0;d=Frb.QV(e);if(d==0){return f;}g=Grb.BW(this,Frb.PV(e,0));Erb.k7(g.d,psb)||(f[b++]=g);for(c=1;c<d;c++){f[b++]=Grb.BW(this,Frb.PV(e,c));}return f;};Grb.GW=function GW(){};Z3(256,377,{},Grb.GW);_.Zb=function HW(a,b,c,d){return new Erb.$6(b,a,-1);};Drb.mX=function mX(a){return a;};function nX(a,b){return Drb.oX(a,b);}Drb.oX=function oX(a,b){return Erb.G5(a,b);};Drb.pX=function pX(a){return a.__elementTypeCategory$==null?10:a.__elementTypeCategory$;};Drb.qX=function qX(a){return a.__elementTypeId$;};function rX(a,b,c,d,e,f){return Drb.sX(a,b,c,d,e,0,f);}Drb.sX=function sX(a,b,c,d,e,f,g){var h,i,j,k,l;k=e[f];j=f==g-1;h=j?d:0;l=Drb.uX(h,k);d!=10&&BX(nX(a,g-f),b[f],c[f],h,l);if(!j){++f;for(i=0;i<k;++i){Drb.wX(l,i,Drb.sX(a,b,c,d,e,f,g));}}return l;};function tX(a,b,c,d,e,f){var g;g=Drb.uX(e,d);e!=10&&BX(nX(a,f),b,c,e,g);return g;}Drb.uX=function uX(a,b){var c=new Array(b);var d;switch(a){case 14:case 15:d=0;break;case 16:d=false;break;default:return c;}for(var e=0;e<b;++e){c[e]=d;}return c;};function vX(a){return Array.isArray(a)&&a.Dc===f4;}Drb.wX=function wX(a,b,c){return a[b]=c;};function xX(a,b,c){return Drb.wX(a,b,c);}Drb.yX=function yX(a,b){a.Bc=b;};Drb.zX=function zX(a,b){a.__elementTypeCategory$=b;};Drb.AX=function AX(a,b){a.__elementTypeId$=b;};function BX(a,b,c,d,e){Drb.yX(e,a);e.Cc=b;e.Dc=f4;Drb.AX(e,c);Drb.zX(e,d);return e;}Drb.CX=function CX(a,b){Drb.pX(b)!=10&&BX(Erb.Nc(b),b.Cc,Drb.qX(b),Drb.pX(b),a);return Drb.mX(a);};function xY(a,b){if(EY(a)){return !!wY[b];}else if(a.Cc){return !!a.Cc[b];}else if(BY(a)){return !!vY[b];}else if(AY(a)){return !!uY[b];}return false;}function yY(a){return !Array.isArray(a)&&a.Dc===f4;}function zY(a,b){return a!=null&&xY(a,b);}function AY(a){return typeof a===usb;}function BY(a){return typeof a===vsb;}function CY(a){return a!=null&&Drb.FY(a)&&!(a.Dc===f4);}function DY(a,b){return Drb.GY(a,b);}function EY(a){return typeof a===wsb;}Drb.FY=function FY(a){return typeof a===osb||typeof a==='function';};Drb.GY=function GY(a,b){return a&&b&&a instanceof b;};function HY(a){return a==null?null:a;}function IY(a){return Math.max(Math.min(a,xsb),-2147483648)|0;}var uY;var vY;var wY;Drb.f3=function f3(a){return a&&a.__java$exception;};function g3(a){var b;if(zY(a,23)){return a;}b=Drb.f3(a);if(!b){b=new Frb.FV(a);Grb.pW(b);}return b;}function h3(a){return a.backingJsObject;}function x4(a){var b;if(a==null){return false;}b=typeof a;return Erb.k7(b,usb)||Erb.k7(b,vsb)||Erb.k7(b,wsb)||a.$implements__java_io_Serializable||Array.isArray(a);}function R5(a){var b;b=typeof a;if(Erb.k7(b,usb)||Erb.k7(b,vsb)||Erb.k7(b,wsb)){return true;}return a!=null&&a.$implements__java_lang_Comparable;}Erb.Q4=function Q4(){Erb.Q4=_3;P4=Hrb.rnb(false);Hrb.rnb(true);};Erb.R4=function R4(a){return Hrb.snb((Hrb.gnb(a),a));};Erb.S4=function S4(a,b){return Y4(Hrb.snb((Hrb.gnb(a),a)),Hrb.snb((Hrb.gnb(b),b)));};Erb.T4=function T4(a,b){return Erb.S4(a,b);};Erb.U4=function U4(a,b){return Hrb.gnb(a),HY(a)===HY(b);};Erb.V4=function V4(){return Drb.f0;};Erb.W4=function W4(a){return Hrb.snb((Hrb.gnb(a),a))?1231:1237;};function X4(a){Erb.Q4();return Erb.k7(usb,typeof a);}function Y4(a,b){Erb.Q4();return a==b?0:a?1:-1;}Erb.Z4=function Z4(a,b){Erb.Q4();return EY(a)?Erb.d7(a,b):BY(a)?Erb.Z5(a,b):AY(a)?Erb.T4(a,b):a.rb(b);};uY={4:1,249:1,28:1};var P4;function $4(a){if(Erb.k7(typeof a,wsb)){return true;}return a!=null&&a.$implements__java_lang_CharSequence;}Erb.t5=function t5(a){a.i=s5++;};Erb.u5=function u5(a){if(a.n!=null){return;}Erb.K5(a);};Erb.v5=function v5(a){Erb.u5(a);return a.n;};Erb.w5=function w5(a){return (a.f&4)!=0;};Erb.x5=function x5(a){return (a.f&1)!=0;};Erb.y5=function y5(){Erb.t5(this);this.n=null;this.j=null;this.g=null;this.d=null;this.b=null;this.k=null;this.a=null;};Erb.A5=function A5(a){var b;b=new Erb.y5();b.n='Class$'+(a?'S'+a:''+b.i);b.b=b.n;b.j=b.n;return b;};function B5(a){var b;b=Erb.A5(a);Erb.O5(a,b);return b;}function C5(a,b){var c;c=Erb.A5(a);Erb.O5(a,c);c.f=b?8:0;return c;}function D5(){var a;a=Erb.A5(null);a.f=2;return a;}function E5(a){var b;b=Erb.A5(a);b.k=a;b.f=1;return b;}Erb.G5=function G5(a,b){var c=a.a=a.a||[];return c[b]||(c[b]=a.$b(b));};function I5(a){if(a.dc()){return null;}var b=a.k;return W3[b];}Erb.K5=function K5(a){if(a.cc()){var b=a.c;b.dc()?a.n='['+b.k:!b.cc()?a.n='[L'+b.ac()+';':a.n='['+b.ac();a.b=b._b()+'[]';a.j=b.bc()+'[]';return;}var c=a.g;var d=a.d;d=d.split('/');a.n=Erb.N5('.',[c,Erb.N5('$',d)]);a.b=Erb.N5('.',[c,Erb.N5('.',d)]);a.j=d[d.length-1];};Erb.N5=function N5(a,b){var c=0;while(!b[c]||b[c]==''){c++;}var d=b[c++];for(;c<b.length;c++){if(!b[c]||b[c]==''){continue;}d+=a+b[c];}return d;};Erb.O5=function O5(a,b){if(!a){return;}b.k=a;var d=I5(b);if(!d){W3[a]=[b];return;}d.Bc=b;};Z3(173,1,{},Erb.y5);_.$b=function z5(a){var b;b=new Erb.y5();b.f=4;a>1?b.c=Erb.G5(this,a-1):b.c=this;return b;};_._b=function F5(){Erb.u5(this);return this.b;};_.ac=function H5(){return Erb.v5(this);};_.bc=function J5(){Erb.u5(this);return this.j;};_.cc=function L5(){return Erb.w5(this);};_.dc=function M5(){return Erb.x5(this);};_.mb=function P5(){return ((this.f&2)!=0?'interface ':(this.f&1)!=0?'':'class ')+(Erb.u5(this),this.n);};_.f=0;_.i=0;var s5=1;Erb.T5=function T5(a){return Erb.k7(vsb,typeof a)||DY(a,$wnd.java.lang.Number$impl);};Erb.U5=function U5(a){Erb.S5==null&&(Erb.S5=new RegExp('^\\s*[+-]?(NaN|Infinity|((\\d+\\.?\\d*)|(\\.\\d+))([eE][+-]?\\d+)?[dDfF]?)\\s*$'));if(!Erb.S5.test(a)){throw h3(new Erb.Z6(ysb+a+'"'));}return parseFloat(a);};Erb.V5=function V5(a,b,c){var d,e,f,g,h;if(a==null){throw h3(new Erb.Z6(zsb));}f=Hrb.rnb(a).length;g=f>0&&(Hrb.mnb(0,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(0)==45||(Hrb.mnb(0,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(0)==43))?1:0;for(d=g;d<f;d++){if(e5((Hrb.mnb(d,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(d)))==-1){throw h3(new Erb.Z6(ysb+a+'"'));}}h=parseInt(a,10);e=h<b;if(isNaN(h)){throw h3(new Erb.Z6(ysb+a+'"'));}else if(e||h>c){throw h3(new Erb.Z6(ysb+a+'"'));}return h;};Erb.W5=function W5(a){return BY(a)?Erb.$5(a):a.ec();};Erb.X5=function X5(a){return BY(a)?Erb.c6(a):a.fc();};Z3(104,1,{4:1,104:1});Erb.Y5=function Y5(a,b){return e6(Hrb.tnb((Hrb.gnb(a),a)),Hrb.tnb((Hrb.gnb(b),b)));};Erb.Z5=function Z5(a,b){return Erb.Y5(a,b);};Erb.$5=function $5(a){return Hrb.tnb((Hrb.gnb(a),a));};Erb._5=function _5(a,b){return Hrb.gnb(a),HY(a)===HY(b);};Erb.a6=function a6(){return Drb.i0;};Erb.b6=function b6(a){return IY(Hrb.tnb((Hrb.gnb(a),a)));};Erb.c6=function c6(a){return IY(Hrb.tnb((Hrb.gnb(a),a)));};function d6(a){return Erb.k7(vsb,typeof a);}function e6(a,b){if(a<b){return  -1;}if(a>b){return 1;}if(a==b){return a==0?e6(1/a,1/b):0;}return isNaN(a)?isNaN(b)?0:1:-1;}function f6(a){if(isNaN(a)){return {l:0,m:0,h:524160};}return Hrb.nnb(a);}vY={4:1,28:1,250:1,104:1};Erb.Dz=function Dz(a){a.g=tX(Drb.x0,ssb,75,0,0,1);};Erb.Ez=function Ez(a){Grb.pW(a);};Erb.Fz=function Fz(a){return Grb.qW(a);};Erb.Gz=function Gz(a){if(a.j){a.backingJsObject!==Asb&&a.Kb();a.g=null;}return a;};Erb.Hz=function Hz(a){return a.e;};Erb.Iz=function Iz(d,b){if(b instanceof Object){try{b.__java$exception=d;if(navigator.userAgent.toLowerCase().indexOf('msie')!=-1&&$doc.documentMode<9){return;}var c=d;Object.defineProperties(b,{cause:{get:function(){var a=c.Ib();return a&&a.Gb();}},suppressed:{get:function(){return c.Hb();}}});}catch(a){}}};Erb.Jz=function Jz(a,b,c){var d,e,f,g,h;Erb.Kz(a);for(e=(a.i==null&&(a.i=tX(Drb.C0,ssb,23,0,0,1)),a.i),f=0,g=e.length;f<g;++f){d=e[f];Erb.Jz(d,b,'\t'+c);}h=a.e;!!h&&Erb.Jz(h,b,c);};Erb.Kz=function Kz(a){var b,c,d;for(b=(a.g==null&&(a.g=Erb.Fz(a)),a.g),c=0,d=b.length;c<d;++c){}};Erb.Lz=function Lz(a,b){a.backingJsObject=b;Erb.Iz(a,b);};Erb.Mz=function Mz(a){return Erb.Nz(a,a.Jb());};Erb.Nz=function Nz(a,b){var c;c=Erb.v5(a.Bc);return b==null?c:c+': '+b;};Erb.Oz=function Oz(){Erb.Dz(this);Erb.Gz(this);this.Kb();};Erb.Pz=function Pz(a){Erb.Dz(this);this.f=a;Erb.Gz(this);this.Kb();};Erb.Rz=function Rz(b){if(!('stack'in b)){try{throw b;}catch(a){}}return b;};function Xz(a){var b;if(a!=null){b=a.__java$exception;if(b){return b;}}return DY(a,TypeError)?new Erb.W6(a):new Erb.xV(a);}Z3(23,1,{4:1,23:1});_.Fb=function Qz(a){return new Error(a);};_.Gb=function Sz(){return this.backingJsObject;};_.Hb=function Tz(){var a,b,c;c=(this.i==null&&(this.i=tX(Drb.C0,ssb,23,0,0,1)),this.i);b=tX(Drb.v0,Bsb,1,c.length,5,1);for(a=0;a<c.length;a++){b[a]=c[a].backingJsObject;}return b;};_.Ib=function Uz(){return Erb.Hz(this);};_.Jb=function Vz(){return this.f;};_.Kb=function Wz(){Erb.Lz(this,Erb.Rz(this.Fb(Erb.Nz(this,this.f))));Erb.Ez(this);};_.mb=function Yz(){return Erb.Mz(this);};_.backingJsObject=Asb;_.j=true;Erb.Zz=function Zz(){Erb.Oz.call(this);};Erb.$z=function $z(a){Erb.Dz(this);Erb.Gz(this);this.backingJsObject=a;Erb.Iz(this,a);this.f=a==null?zsb:e4(a);};Erb._z=function _z(a){Erb.Pz.call(this,a);};Z3(15,23,{4:1,15:1,23:1},Erb._z);Erb.aA=function aA(){Erb.Zz.call(this);};Erb.bA=function bA(a){Erb.$z.call(this,a);};Erb.cA=function cA(a){Erb._z.call(this,a);};Z3(26,15,Csb);Erb.L4=function L4(){Erb.aA.call(this);};Erb.M4=function M4(a){Erb.cA.call(this,a);};Z3(68,26,Csb,Erb.L4,Erb.M4);Erb.wV=function wV(){Erb.aA.call(this);};Erb.xV=function xV(a){Erb.bA.call(this,a);};Erb.yV=function yV(a){Erb.cA.call(this,a);};Z3(109,26,Csb,Erb.xV);Erb.V6=function V6(){Erb.wV.call(this);};Erb.W6=function W6(a){Erb.xV.call(this,a);};Erb.X6=function X6(a){Erb.yV.call(this,a);};Z3(63,109,Csb,Erb.V6,Erb.W6,Erb.X6);_.Fb=function Y6(a){return new TypeError(a);};Erb.c7=function c7(a,b){Hrb.mnb(b,Hrb.rnb(a).length);return Hrb.rnb(a).charCodeAt(b);};Erb.d7=function d7(a,b){return Erb.e7(a,b);};Erb.e7=function e7(a,b){var c,d;c=Erb.$5(Hrb.rnb(a));d=Erb.$5(Hrb.rnb(b));return c==d?0:c<d?-1:1;};Erb.f7=function f7(a,b){return Erb.e7(Hrb.rnb(a).toLowerCase(),Hrb.rnb(b).toLowerCase());};Erb.g7=function g7(a,b){return Hrb.gnb(a),a+(''+(Hrb.gnb(b),b));};Erb.h7=function h7(a){var b;return Erb.D7(Hrb.Smb(a,0,(b=a.length,Hrb.Pmb(),b)));};Erb.i7=function i7(a){return Erb.D7(Hrb.Smb(a,0,a.length));};Erb.j7=function j7(a){var b;b=Hrb.rnb(Dsb).length;return Erb.k7(Hrb.rnb(a).substr(Hrb.rnb(a).length-b,b),Dsb);};Erb.k7=function k7(a,b){return Hrb.gnb(a),HY(a)===HY(b);};Erb.l7=function l7(a,b){Hrb.gnb(a);if(b==null){return false;}if(Erb.k7(a,b)){return true;}return Hrb.rnb(a).length==Hrb.rnb(b).length&&Erb.k7(Hrb.rnb(a).toLowerCase(),Hrb.rnb(b).toLowerCase());};Erb.m7=function m7(){return Drb.B0;};Erb.n7=function n7(a){var b,c;b=0;for(c=0;c<Hrb.rnb(a).length;c++){b=(b<<5)-b+(Hrb.mnb(c,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(c))|0;}return b;};Erb.o7=function o7(a,b){return Hrb.rnb(a).indexOf(b);};Erb.p7=function p7(a,b,c){return Hrb.rnb(a).indexOf(b,c);};function q7(a){return Erb.k7(wsb,typeof a);}Erb.r7=function r7(a,b){return Hrb.rnb(a).lastIndexOf(b);};Erb.s7=function s7(a,b,c){return Hrb.rnb(a).lastIndexOf(b,c);};Erb.t7=function t7(a,b){var c;c=Erb.u7(b,'([/\\\\\\.\\*\\+\\?\\|\\(\\)\\[\\]\\{\\}$^])','\\\\$1');return Erb.u7(a,c,'');};Erb.u7=function u7(a,b,c){c=Erb.C7(c);return Hrb.rnb(a).replace(new RegExp(b,'g'),c);};Erb.v7=function v7(a,b){var c,d,e,f,g,h,i,j;c=new RegExp(b,'g');i=tX(Drb.B0,Esb,2,0,6,1);d=0;j=a;f=null;while(true){h=c.exec(j);if(h==null||j==''){i[d]=j;break;}else {g=h.index;i[d]=(Hrb.lnb(0,g,Hrb.rnb(j).length),Hrb.rnb(j).substr(0,g));j=Erb.x7(j,g+Hrb.rnb(Hrb.rnb(h)[0]).length,Hrb.rnb(j).length);c.lastIndex=0;if(f==j){i[d]=(Hrb.lnb(0,1,Hrb.rnb(j).length),Hrb.rnb(j).substr(0,1));j=(Hrb.mnb(1,Hrb.rnb(j).length+1),Hrb.rnb(j).substr(1));}f=j;++d;}}if(Hrb.rnb(a).length>0){e=i.length;while(e>0&&i[e-1]==''){--e;}e<i.length&&(Hrb.rnb(i).length=e);}return i;};Erb.w7=function w7(a,b){Hrb.mnb(b,Hrb.rnb(a).length+1);return Hrb.rnb(a).substr(b);};Erb.x7=function x7(a,b,c){Hrb.lnb(b,c,Hrb.rnb(a).length);return Hrb.rnb(a).substr(b,c-b);};Erb.y7=function y7(a){var b,c,d;c=Hrb.rnb(a).length;d=0;while(d<c&&(Hrb.mnb(d,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(d)<=32)){++d;}b=c;while(b>d&&(Hrb.mnb(b-1,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(b-1)<=32)){--b;}return d>0||b<c?(Hrb.lnb(d,b,Hrb.rnb(a).length),Hrb.rnb(a).substr(d,b-d)):a;};Erb.z7=function z7(a){return String.fromCharCode.apply(null,a);};function A7(a){var b,c;if(a>=Fsb){b=55296+(a-Fsb>>10&1023)&Gsb;c=56320+(a-Fsb&1023)&Gsb;return String.fromCharCode(b)+(''+String.fromCharCode(c));}else {return String.fromCharCode(a&Gsb);}}Erb.B7=function B7(a){var b,c,d,e,f;f=new Krb.ejb();for(c=a,d=0,e=c.length;d<e;++d){b=c[d];!f.a?f.a=new Erb.U7(f.d):Erb.P7(f.a,f.b);Erb.O7(f.a,b);}return !f.a?f.c:Hrb.rnb(f.e).length==0?f.a.a:f.a.a+(''+f.e);};Erb.C7=function C7(a){var b;b=0;while(0<=(b=Hrb.rnb(a).indexOf('\\',b))){Hrb.mnb(b+1,Hrb.rnb(a).length);Hrb.rnb(a).charCodeAt(b+1)==36?a=(Hrb.lnb(0,b,Hrb.rnb(a).length),Hrb.rnb(a).substr(0,b)+'$'+Erb.w7(a,++b)):a=(Hrb.lnb(0,b,Hrb.rnb(a).length),Hrb.rnb(a).substr(0,b)+(''+Erb.w7(a,++b)));}return a;};Erb.D7=function D7(a){return Erb.E7(a,0,a.length);};Erb.E7=function E7(a,b,c){var d,e,f,g;f=b+c;Hrb.lnb(b,f,a.length);g='';for(e=b;e<f;){d=$wnd.Math.min(e+Hsb,f);g+=Erb.z7(Hrb.rnb(a).slice(e,d));e=d;}return g;};wY={4:1,175:1,28:1,2:1};Erb.V7=function V7(a){Erb.M4.call(this,a);};Z3(196,68,Csb,Erb.V7);Hrb.Emb=function Emb(a){var b;b=Hrb.rnb(a).slice();return Hrb.Omb(b,a);};Hrb.Fmb=function Fmb(a,b,c,d,e,f){var g,h,i,j,k;if(e==0){return;}if(HY(a)===HY(c)){a=Hrb.rnb(a).slice(b,b+e);b=0;}i=Hrb.rnb(c);for(h=b,j=b+e;h<j;){g=$wnd.Math.min(h+Hsb,j);e=g-h;k=Hrb.rnb(a).slice(h,g);Hrb.rnb(k).splice(0,0,Hrb.rnb(d),Hrb.rnb(f?e:0));Array.prototype.splice.apply(i,k);h=g;d+=e;}};Hrb.Gmb=function Gmb(a,b){return Hrb.Omb(new Array(b),a);};Hrb.Hmb=function Hmb(a,b,c){Hrb.rnb(a).splice(b,0,c);};Hrb.Imb=function Imb(a,b,c){Hrb.Fmb(c,0,a,b,c.length,false);};Hrb.Jmb=function Jmb(a,b){Hrb.rnb(a).push(b);};Hrb.Kmb=function Kmb(a,b){Hrb.rnb(a).push(b);};Hrb.Lmb=function Lmb(a,b,c){Hrb.rnb(a).splice(b,c);};Hrb.Mmb=function Mmb(a,b){Hrb.rnb(a).length=b;};Hrb.Nmb=function Nmb(a,b){Hrb.rnb(a).sort(b);};Z3(429,1,{});Hrb.Xmb=function Xmb(){};Hrb.Ymb=function Ymb(a){switch(typeof a){case wsb:return Erb.n7(Hrb.rnb(a));case vsb:return Erb.b6(Hrb.rnb(a));case usb:return Erb.W4(Hrb.rnb(a));default:return a==null?0:Hrb.$mb(a);}};Hrb.Zmb=function Zmb(){return ++Hrb.Wmb;};Hrb.$mb=function $mb(a){return a.$H||(a.$H=Hrb.Zmb());};Z3(374,1,{},Hrb.Xmb);Hrb.Wmb=0;Hrb._mb=function _mb(a){if(!a){throw h3(new Erb.h6());}};Hrb.anb=function anb(a,b){if(!a){throw h3(new Erb.i6(b));}};Hrb.bnb=function bnb(a,b,c){if(a>b){throw h3(new Erb.i6(Isb+a+Jsb+b));}if(a<0||b>c){throw h3(new Erb.O4(Isb+a+Ksb+b+Lsb+c));}};Hrb.cnb=function cnb(a){if(a<0){throw h3(new Erb.U6('Negative array size: '+a));}};Hrb.dnb=function dnb(a,b){if(a!=b){throw h3(new Krb.qhb());}};Hrb.enb=function enb(a){if(!a){throw h3(new Krb.Jib());}};Hrb.fnb=function fnb(a,b){if(a<0||a>=b){throw h3(new Erb.M4(Msb+a+Nsb+b));}};Hrb.gnb=function gnb(a){if(a==null){throw h3(new Erb.V6());}return a;};Hrb.hnb=function hnb(a,b){if(a==null){throw h3(new Erb.X6(b));}};Hrb.inb=function inb(a,b){if(a<0||a>b){throw h3(new Erb.M4(Msb+a+Nsb+b));}};Hrb.jnb=function jnb(a,b,c){if(a<0||b>c){throw h3(new Erb.M4(Isb+a+Ksb+b+', size: '+c));}if(a>b){throw h3(new Erb.i6(Isb+a+Jsb+b));}};Hrb.knb=function knb(a){if(!a){throw h3(new Erb.j6());}};Hrb.lnb=function lnb(a,b,c){if(a<0||b>c||b<a){throw h3(new Erb.V7(Isb+a+Ksb+b+Lsb+c));}};Hrb.mnb=function mnb(a,b){if(a<0||a>=b){throw h3(new Erb.V7(Msb+a+Nsb+b));}};Hrb.nnb=function nnb(a){var b,c;b=new ArrayBuffer(8);Hrb.rnb(new Float64Array(b))[0]=a;c=Hrb.rnb(new Uint32Array(b));return Hrb.unb(c[0]|0,c[1]|0);};Hrb.onb=function onb(a,b){return a[b];};Hrb.pnb=function pnb(a){return a===undefined;};Hrb.qnb=function qnb(a){return a>>>0;};Hrb.rnb=function rnb(a){return a;};Hrb.snb=function snb(a){return a;};Hrb.tnb=function tnb(a){return a;};Z3(375,1,{});Drb.v0=B5(1);Drb.H_=B5(0);Drb.O_=B5(376);Drb.L_=B5(255);Drb.N_=B5(377);Drb.M_=B5(256);Drb.f0=B5(249);Drb.h0=B5(173);Drb.u0=B5(104);Drb.i0=B5(250);Drb.C0=B5(23);Drb.k0=B5(15);Drb.w0=B5(26);Drb.n0=B5(68);Drb.p0=B5(109);Drb.s0=B5(63);Drb.B0=B5(2);Drb.A0=B5(196);Drb.y2=B5(374);Drb.z2=B5(375);Irb.Sc=function Sc(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q;this.c=a.length;this.d=a[0].length;G=$wnd.Math.min(this.c,this.d);this.e=tX(Drb.LY,Osb,6,$wnd.Math.min(this.c+1,this.d),15,1);this.a=rX(Drb.LY,[ssb,Osb],[12,6],15,[this.c,G],2);this.b=rX(Drb.LY,[ssb,Osb],[12,6],15,[this.d,this.d],2);e=tX(Drb.LY,Osb,6,this.d,15,1);Q=tX(Drb.LY,Osb,6,this.c,15,1);D=$wnd.Math.min(this.c-1,this.d);F=$wnd.Math.max(0,$wnd.Math.min(this.d-2,this.c));for(v=0;v<$wnd.Math.max(D,F);v++){if(v<D){this.e[v]=0;for(l=v;l<this.c;l++){this.e[v]=Irb.Tc(this.e[v],a[l][v]);}if(this.e[v]!=0){a[v][v]<0&&(this.e[v]=-this.e[v]);for(k=v;k<this.c;k++){a[k][v]/=this.e[v];}a[v][v]+=1;}this.e[v]=-this.e[v];}for(q=v+1;q<this.d;q++){if(v<D&this.e[v]!=0){P=0;for(l=v;l<this.c;l++){P+=a[l][v]*a[l][q];}P=-P/a[v][v];for(k=v;k<this.c;k++){a[k][q]+=P*a[k][v];}}e[q]=a[v][q];}if(true&v<D){for(k=v;k<this.c;k++){this.a[k][v]=a[k][v];}}if(v<F){e[v]=0;for(l=v+1;l<this.d;l++){e[v]=Irb.Tc(e[v],e[l]);}if(e[v]!=0){e[v+1]<0&&(e[v]=-e[v]);for(m=v+1;m<this.d;m++){e[m]/=e[v];}e[v+1]+=1;}e[v]=-e[v];if(v+1<this.c&e[v]!=0){for(m=v+1;m<this.c;m++){Q[m]=0;}for(r=v+1;r<this.d;r++){for(n=v+1;n<this.c;n++){Q[n]+=e[r]*a[n][r];}}for(s=v+1;s<this.d;s++){P=-e[s]/e[v+1];for(n=v+1;n<this.c;n++){a[n][s]+=P*Q[n];}}}for(k=v+1;k<this.d;k++){this.b[k][v]=e[k];}}}H=$wnd.Math.min(this.d,this.c+1);D<this.d&&(this.e[D]=a[D][D]);this.c<H&&(this.e[H-1]=0);F+1<H&&(e[F]=a[F][H-1]);e[H-1]=0;for(t=D;t<G;t++){for(k=0;k<this.c;k++){this.a[k][t]=0;}this.a[t][t]=1;}for(w=D-1;w>=0;w--){if(this.e[w]!=0){for(p=w+1;p<G;p++){P=0;for(l=w;l<this.c;l++){P+=this.a[l][w]*this.a[l][p];}P=-P/this.a[w][w];for(m=w;m<this.c;m++){this.a[m][p]+=P*this.a[m][w];}}for(n=w;n<this.c;n++){this.a[n][w]=-this.a[n][w];}this.a[w][w]=1+this.a[w][w];for(k=0;k<w-1;k++){this.a[k][w]=0;}}else {for(k=0;k<this.c;k++){this.a[k][w]=0;}this.a[w][w]=1;}}for(A=this.d-1;A>=0;A--){if(A<F&e[A]!=0){for(p=A+1;p<G;p++){P=0;for(l=A+1;l<this.d;l++){P+=this.b[l][A]*this.b[l][p];}P=-P/this.b[A+1][A];for(m=A+1;m<this.d;m++){this.b[m][p]+=P*this.b[m][A];}}}for(k=0;k<this.d;k++){this.b[k][A]=0;}this.b[A][A]=1;}I=H-1;h=$wnd.Math.pow(2,-52);while(H>0){for(u=H-2;u>=-1;u--){if(u==-1){break;}if($wnd.Math.abs(e[u])<=h*($wnd.Math.abs(this.e[u])+$wnd.Math.abs(this.e[u+1]))){e[u]=0;break;}}if(u==H-2){B=4;}else {for(C=H-1;C>=u;C--){if(C==u){break;}P=(C!=H?$wnd.Math.abs(e[C]):0)+(C!=u+1?$wnd.Math.abs(e[C-1]):0);if($wnd.Math.abs(this.e[C])<=h*P){this.e[C]=0;break;}}if(C==u){B=3;}else if(C==H-1){B=1;}else {B=2;u=C;}}++u;switch(B){case 1:{i=e[H-2];e[H-2]=0;for(p=H-2;p>=u;p--){P=Irb.Tc(this.e[p],i);d=this.e[p]/P;M=i/P;this.e[p]=P;if(p!=u){i=-M*e[p-1];e[p-1]=d*e[p-1];}for(k=0;k<this.d;k++){P=d*this.b[k][p]+M*this.b[k][H-1];this.b[k][H-1]=-M*this.b[k][p]+d*this.b[k][H-1];this.b[k][p]=P;}}}break;case 2:{i=e[u-1];e[u-1]=0;for(p=u;p<H;p++){P=Irb.Tc(this.e[p],i);d=this.e[p]/P;M=i/P;this.e[p]=P;i=-M*e[p];e[p]=d*e[p];for(k=0;k<this.c;k++){P=d*this.a[k][p]+M*this.a[k][u-1];this.a[k][u-1]=-M*this.a[k][p]+d*this.a[k][u-1];this.a[k][p]=P;}}}break;case 3:{J=$wnd.Math.max($wnd.Math.max($wnd.Math.max($wnd.Math.max($wnd.Math.abs(this.e[H-1]),$wnd.Math.abs(this.e[H-2])),$wnd.Math.abs(e[H-2])),$wnd.Math.abs(this.e[u])),$wnd.Math.abs(e[u]));N=this.e[H-1]/J;O=this.e[H-2]/J;g=e[H-2]/J;L=this.e[u]/J;f=e[u]/J;b=((O+N)*(O-N)+g*g)/2;c=N*g*(N*g);K=0;if(b!=0|c!=0){K=$wnd.Math.sqrt(b*b+c);b<0&&(K=-K);K=c/(b+K);}i=(L+N)*(L-N)+K;j=L*f;for(p=u;p<H-1;p++){P=Irb.Tc(i,j);d=i/P;M=j/P;p!=u&&(e[p-1]=P);i=d*this.e[p]+M*e[p];e[p]=d*e[p]-M*this.e[p];j=M*this.e[p+1];this.e[p+1]=d*this.e[p+1];for(l=0;l<this.d;l++){P=d*this.b[l][p]+M*this.b[l][p+1];this.b[l][p+1]=-M*this.b[l][p]+d*this.b[l][p+1];this.b[l][p]=P;}P=Irb.Tc(i,j);d=i/P;M=j/P;this.e[p]=P;i=d*e[p]+M*this.e[p+1];this.e[p+1]=-M*e[p]+d*this.e[p+1];j=M*e[p+1];e[p+1]=d*e[p+1];if(p<this.c-1){for(k=0;k<this.c;k++){P=d*this.a[k][p]+M*this.a[k][p+1];this.a[k][p+1]=-M*this.a[k][p]+d*this.a[k][p+1];this.a[k][p]=P;}}}e[H-2]=i;}break;case 4:{if(this.e[u]<=0){this.e[u]=this.e[u]<0?-this.e[u]:0;for(k=0;k<=I;k++){this.b[k][u]=-this.b[k][u];}}while(u<I){if(this.e[u]>=this.e[u+1]){break;}P=this.e[u];this.e[u]=this.e[u+1];this.e[u+1]=P;if(u<this.d-1){for(k=0;k<this.d;k++){P=this.b[k][u+1];this.b[k][u+1]=this.b[k][u];this.b[k][u]=P;}}if(u<this.c-1){for(k=0;k<this.c;k++){P=this.a[k][u+1];this.a[k][u+1]=this.a[k][u];this.a[k][u]=P;}}++u;}--H;}}}};Irb.Tc=function Tc(a,b){var c;if($wnd.Math.abs(a)>$wnd.Math.abs(b)){c=b/a;c=$wnd.Math.abs(a)*$wnd.Math.sqrt(1+c*c);}else if(b!=0){c=a/b;c=$wnd.Math.abs(b)*$wnd.Math.sqrt(1+c*c);}else {c=0;}return c;};Z3(171,1,Psb,Irb.Sc);_.c=0;_.d=0;Drb.PY=B5(171);Jrb.Vc=function Vc(){Jrb.Vc=_3;Jrb.Uc=BX(nX(Drb.NY,1),Qsb,6,15,[0,Rsb,14286847,13402367,12779264,16758197,9474192,3166456,16715021,9494608,11789301,11230450,9109248,12560038,15780000,16744448,16777008,2093087,8442339,9388244,4062976,15132390,12567239,10921643,9083335,10255047,14706227,15765664,5296208,13140019,8224944,12750735,6721423,12419299,16752896,10889513,6076625,7351984,Ssb,9764863,9756896,7586505,5551541,3907230,2396047,687500,27013,12632256,16767375,10909043,6717568,10380213,13924864,9699476,4366000,5707663,51456,7394559,16777159,14286791,13107143,10747847,9437127,6422471,4587463,3211207,2097095,65436,58997,54354,48952,43812,5096191,5089023,2200790,2522539,2516630,1528967,13684960,16765219,12105936,10900557,5724513,10375093,11230208,7688005,4358806,4325478,32000,7384058,47871,41471,36863,33023,27647,5528818,7888099,9064419,10565332,11739092,11739066,11734438,12389767,13041766,13369433,13697103,14221381,14680120,15073326,15400998,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13158600,1334015,56540,15075850,15132160,56540,15075850,15461355,8553170,1016335,1016335,1334015,15132160,3289770,14456450,16422400,16422400,11819700,3289770,1016335]);};Jrb.Wc=function Wc(a){a.v=new Wrb.KG();};Jrb.Xc=function Xc(a){var b,c;if((a.F&32)!=0)return;c=Jrb.ru(a.K);if(c!=null){if(a.w.a==0&&a.w.b==0){b=a.Q.c*Jrb.Ek(a.K);a.v=Jrb.Qd(a);Jrb.gd(a,b);Jrb.Nd(a,null,b,0);}Jrb.at(a,IY(a.A));a.K.O!=1&&Jrb.Od(a,448);Jrb.Ws(a,c,a.w.a,a.w.b+Tsb*a.A);}};Jrb.Yc=function Yc(a,b){return a==null?b:b==null?a:a+','+b;};Jrb.Zc=function Zc(a){var b;b=a.Q.c*(a.r!=0?a.r:Jrb.Ek(a.K));a.X=b*0.06;a.S=b*0.15;a.R=b*0.38;a.V=b*0.47;a.W=IY(b*a.I*0.6+0.5);a.U=b*0.12;a.Y=b*0.4;a.A=b*0.5+0.5;};Jrb.$c=function $c(a){var b,c;a=j3(a,Usb);for(c=0;c<(Vrb.AG(),Vrb.zG).length;c++)if(t3(a,Vrb.zG[c]))return Vrb.yG[c];b=new Erb.U7('R');E3(j3(a,Vsb),0)&&(b.a+='0',b);E3(j3(a,Wsb),0)&&(b.a+='3',b);E3(j3(a,Xsb),0)&&(b.a+='4',b);E3(j3(a,Ysb),0)&&(b.a+='5',b);E3(j3(a,Zsb),0)&&(b.a+='6',b);E3(j3(a,$sb),0)&&(b.a+='7',b);E3(j3(a,_sb),0)&&(b.a+='8',b);return b.a;};Jrb._c=function _c(a,b,c,d){var e,f,g;e=new Jrb.Wd();f=new Jrb.Wd();e.a=b.a;e.c=b.c;e.b=(b.a+b.b)/2;e.d=(b.c+b.d)/2;f.a=e.b;f.c=e.d;f.b=b.b;f.d=b.d;if(Jrb.Kd(a,e)){Jrb.Od(a,a.p[c]);g=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,g);}if(Jrb.Kd(a,f)){Jrb.Od(a,a.p[d]);g=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,g);}Jrb.Od(a,a.P);};Jrb.ad=function ad(a,b,c,d){var e,f,g,h,i,j,k,l,m;l=(b.b-b.a)/10;m=(b.d-b.c)/10;e=new Jrb.Wd();if(Jrb.ml(a.K,Jrb.Dn(a.K,c,d))){f=-3;g=-3;}else {f=a.p[c];g=a.p[d];}Jrb.Od(a,f);e.a=b.a;e.c=b.c;e.b=b.a+l*2;e.d=b.c+m*2;i=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,i);e.a=b.a+l*4;e.c=b.c+m*4;e.b=b.a+l*5;e.d=b.c+m*5;j=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.a),gsb.xab)))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,j);Jrb.Od(a,g);e.a=b.a+l*5;e.c=b.c+m*5;e.b=b.a+l*6;e.d=b.c+m*6;k=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.a),gsb.xab)))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,k);e.a=b.a+l*8;e.c=b.c+m*8;e.b=b.b;e.d=b.d;h=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.a),gsb.xab)))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,h);Jrb.Od(a,a.P);};Jrb.bd=function bd(a,b,c){Jrb.Xs(a,b-a.U/2,c-a.U/2,a.U);};Jrb.cd=function cd(a,b,c,d){var e;if(Jrb.ml(a.K,Jrb.Dn(a.K,c,d))){Jrb.Od(a,-3);e=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,e);Jrb.Od(a,a.P);}else if(a.p[c]!=a.p[d]){Jrb._c(a,b,c,d);}else if(a.p[c]!=0){Jrb.Od(a,a.p[c]);e=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,e);Jrb.Od(a,a.P);}else {e=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,e);}};Jrb.dd=function dd(a,b,c,d,e,f,g){g.a=b.a+e;g.c=b.c+f;g.b=b.b+e;g.d=b.d+f;Jrb.cd(a,g,c,d);};Jrb.ed=function ed(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p;l=b.b-b.a;o=b.d-b.c;i=$wnd.Math.sqrt(l*l+o*o);j=2*N3(u3($wnd.Math.round(i/(4*a.X))));m=l/(j-1);p=o/(j-1);if(Jrb.ml(a.K,Jrb.Dn(a.K,c,d))){e=-3;f=-3;}else {e=a.p[c];f=a.p[d];}k=b.a-a.X/2;n=b.c-a.X/2;Jrb.Od(a,e);for(h=0;h<(j/2|0);h++){Jrb.Xs(a,k,n,a.X);k+=m;n+=p;}Jrb.Od(a,f);for(g=0;g<(j/2|0);g++){Jrb.Xs(a,k,n,a.X);k+=m;n+=p;}Jrb.Od(a,a.P);};Jrb.fd=function fd(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r;i=(b.c-b.d)/9;n=(b.b-b.a)/9;j=b.b+i;o=b.d+n;k=b.b-i;p=b.d-n;l=(b.a+j)/2;q=(b.c+o)/2;m=(b.a+k)/2;r=(b.c+p)/2;g=new Wrb.FG(3);h=new Wrb.FG(4);Wrb.EG(g,b.a,b.c);Wrb.EG(g,l,q);Wrb.EG(g,m,r);Wrb.EG(h,m,r);Wrb.EG(h,l,q);Wrb.EG(h,j,o);Wrb.EG(h,k,p);if(Jrb.ml(a.K,Jrb.Dn(a.K,c,d))){e=-3;f=-3;}else {e=a.p[c];f=a.p[d];if(a.K.O!=1){f=Jrb.jd(a,c);e==Jrb.nk(a.K,c)&&(e=f);}}Jrb.Od(a,e);Jrb.Vs(a,g);Jrb.Od(a,f);Jrb.Vs(a,h);Jrb.Od(a,a.P);};Jrb.gd=function gd(a,b){var c,d;for(d=0;d<a.Z.a.length;d++)a.v=Wrb.JG(a.v,Krb.Ji(a.Z,d));Jrb.hd(a,b);c=0.1*b;a.v.c-=c;a.v.d-=c;a.v.b+=2*c;a.v.a+=2*c;};Jrb.hd=function hd(a,b){var c,d,e,f,g,h,i;e=tX(Drb.c3,itb,6,a.K.q,16,1);for(d=0;d<a.K.r;d++){if(Jrb.kl(a.K,d)){e[Jrb.Ik(a.K,0,d)]=true;e[Jrb.Ik(a.K,1,d)]=true;}}g=new Wrb.KG();for(c=0;c<a.K.q;c++){f=E3(j3(Jrb.yk(a.K,c),jtb),0)?b*0.47:e[c]?b*0.38:0;if(f!=0){h=Jrb.Qh(a.Q,Jrb.Ak(a.K,c));i=Jrb.Rh(a.Q,Jrb.Bk(a.K,c));Wrb.IG(g,h-f,i-f,f*2,f*2);a.v=Wrb.JG(a.v,g);}}};Jrb.jd=function jd(a,b){var c,d;if((a.F&4224)!=0)return a.p[b];d=Jrb.kd(a,b);if(d==-1){c=Jrb.ln(a.K,b);if(c!=-1){b=c;d=Jrb.kd(a,b);}}if(d==-1)return a.p[b];switch(d&255){case 1:return 384;case 2:return 64;default:return 448;}};Jrb.kd=function kd(a,b){var c,d,e;e=-1;d=-1;if((a.F&128)!=0)return e;if(Jrb.il(a.K,b)){e=Jrb.rk(a.K,b);d=Jrb.qk(a.K,b);}c=Jrb.nn(a.K,b);if(c!=-1){e=Jrb.Nk(a.K,c);d=Jrb.Mk(a.K,c);}e!=-1&&e!=0&&(e|=d<<8);return e;};Jrb.ld=function ld(a){var b,c,d,e,f;Jrb.$s(a,2*a.R);e=new Jrb.Wd();for(d=0;d<a.K.r;d++){b=Jrb.Ik(a.K,0,d);c=Jrb.Ik(a.K,1,d);if(Jrb.kl(a.K,d)){e.a=Jrb.Qh(a.Q,Jrb.Ak(a.K,b));e.c=Jrb.Rh(a.Q,Jrb.Bk(a.K,b));e.b=Jrb.Qh(a.Q,Jrb.Ak(a.K,c));e.d=Jrb.Rh(a.Q,Jrb.Bk(a.K,c));Jrb.Od(a,-2);f=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,f);}}};Jrb.md=function md(a){var b,c,d,e,f,g,h;if(a.K.K){g=a.V;Jrb.Od(a,-7);for(b=0;b<a.K.f;b++)E3(j3(Jrb.yk(a.K,b),jtb),0)&&Jrb.Xs(a,Jrb.Qh(a.Q,Jrb.Ak(a.K,b))-g,Jrb.Rh(a.Q,Jrb.Bk(a.K,b))-g,2*g);Jrb.$s(a,2*a.V);f=new Jrb.Wd();for(e=0;e<a.K.r;e++){c=Jrb.Ik(a.K,0,e);d=Jrb.Ik(a.K,1,e);if(E3(j3(j3(Jrb.yk(a.K,c),Jrb.yk(a.K,d)),jtb),0)){f.a=Jrb.Qh(a.Q,Jrb.Ak(a.K,c));f.c=Jrb.Rh(a.Q,Jrb.Bk(a.K,c));f.b=Jrb.Qh(a.Q,Jrb.Ak(a.K,d));f.d=Jrb.Rh(a.Q,Jrb.Bk(a.K,d));h=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,h);}}}};Jrb.nd=function nd(a){var b,c,d,e;if(a.K.K){Jrb.Od(a,320);if((a.F&8)!=0)for(b=0;b<a.K.f;b++)E3(j3(Jrb.yk(a.K,b),-536870913),0)&&Jrb.Xs(a,Jrb.Qh(a.Q,Jrb.Ak(a.K,b))-a.Y/2,Jrb.Rh(a.Q,Jrb.Bk(a.K,b))-a.Y/2,a.Y);for(e=0;e<a.K.g;e++){if(Jrb.Sk(a.K,e)!=0){c=Jrb.Ik(a.K,0,e);d=Jrb.Ik(a.K,1,e);Jrb.Xs(a,(Jrb.Qh(a.Q,Jrb.Ak(a.K,c))+Jrb.Qh(a.Q,Jrb.Ak(a.K,d))-a.Y)/2,(Jrb.Rh(a.Q,Jrb.Bk(a.K,c))+Jrb.Rh(a.Q,Jrb.Bk(a.K,d))-a.Y)/2,a.Y);}}}};Jrb.od=function od(a){a.I=1;a.Q=new Jrb.Sh();a.Z=new Krb.Yi();a.T=new Krb.Yi();a.s=tX(Drb.c3,itb,6,a.K.q,16,1);a.w=new Wrb.BG();a.P=0;a.B=-1;Jrb.Sd(a);};Jrb.pd=function pd(a,b){var c;if(Jrb.rn(a.K,b)==0)return false;for(c=0;c<Jrb.rn(a.K,b);c++)if(!Jrb.ml(a.K,Jrb.Hn(a.K,b,c)))return false;return true;};Jrb.qd=function qd(a){var b;a.q=tX(Drb.c3,itb,6,a.K.q,16,1);for(b=0;b<a.K.r;b++){a.q[Jrb.Ik(a.K,0,b)]=true;a.q[Jrb.Ik(a.K,1,b)]=true;}};Jrb.rd=function rd(a,b){var c;if(Jrb.Gn(a.K,b)!=2)return false;for(c=0;c<2;c++)if(Jrb.In(a.K,b,c)!=2)return false;return true;};Jrb.sd=function sd(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o;m=false;e.a=0;e.b=0;d>0?f=ktb:f=ltb;o=Jrb.Hk(a.K,b,c);for(k=0;k<Jrb.Gn(a.K,b);k++){g=Jrb.Hn(a.K,b,k);h=o;Jrb.Ik(a.K,0,g)==b?l=Jrb.Ik(a.K,1,g):l=Jrb.Ik(a.K,0,g);if(l==c)continue;n=Jrb.Hk(a.K,b,l);h<n&&(h+=mtb);i=h-n;if(d>0){i<ntb&&(m=true);i>ktb&&(i=ktb);i<0.523598776&&(i=0.523598776);if(i<=f){f=i;j=a.S*$wnd.Math.tan(f-otb)/2;e.a=-(j*$wnd.Math.sin(h));e.b=-(j*$wnd.Math.cos(h));}}else {i>=ntb&&(m=true);i<ltb&&(i=ltb);i>5.759586531&&(i=5.759586531);if(i>=f){f=i;j=a.S*$wnd.Math.tan(4.712388981-f)/2;e.a=-(j*$wnd.Math.sin(h));e.b=-(j*$wnd.Math.cos(h));}}}return m;};Jrb.td=function td(a,b,c,d){var e;if(b==0){c<0?d.a=a.S:d.a=-a.S;d.b=0;return;}e=$wnd.Math.atan(c/b);b<0&&(e+=ntb);d.a=-(a.S*$wnd.Math.sin(e));d.b=a.S*$wnd.Math.cos(e);};Jrb.ud=function ud(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p;e=new Jrb.Wd();i=new Jrb.Wd();k=new Wrb.BG();j=new Wrb.BG();g=Jrb.Ik(a.K,0,c);h=Jrb.Ik(a.K,1,c);if(d){m=b.a;b.a=b.b;b.b=m;m=b.c;b.c=b.d;b.d=m;n=g;g=h;h=n;}if(!Jrb.Kd(a,b))return;if(Jrb.Mo(a.K,c)){e.a=b.a;e.c=b.c;e.b=b.b;e.d=b.d;l=d?-Jrb.io(a.K,c):Jrb.io(a.K,c);l==0&&(l=1);Jrb.td(a,b.b-b.a,b.d-b.c,k);if(l>0){i.a=b.a+k.a;i.c=b.c+k.b;i.b=b.b+k.a;i.d=b.d+k.b;if(Jrb.sd(a,g,h,1,j)||Jrb.Gn(a.K,g)>1){i.a+=j.a+k.b;i.c+=j.b-k.a;}}else {i.a=b.a-k.a;i.c=b.c-k.b;i.b=b.b-k.a;i.d=b.d-k.b;if(Jrb.sd(a,g,h,-1,j)||Jrb.Gn(a.K,g)>1){i.a+=j.a+k.b;i.c+=j.b-k.a;}}Jrb.Tk(a.K,c)==386&&Jrb.Jd(e,i);Jrb.Kd(a,e)&&Jrb.cd(a,e,g,h);Jrb.Tk(a.K,c)==8?Jrb.Kd(a,i)&&Jrb.ad(a,i,g,h):Jrb.Kd(a,i)&&Jrb.cd(a,i,g,h);}else {Jrb.td(a,b.b-b.a,b.d-b.c,k);o=k.a/2;p=k.b/2;f=false;e.a=b.a+o;e.c=b.c+p;e.b=b.b+o;e.d=b.d+p;if(Jrb.Gn(a.K,g)>1){if(Jrb.sd(a,g,h,1,j)){e.a+=j.a;e.c+=j.b;if(Jrb.Gn(a.K,g)==2){if(j.a!=0||j.b!=0){e.a+=k.b;e.c-=k.a;}}}else {a.o[g]=new Wrb.CG(e.a,e.c);}}i.a=b.a-o;i.c=b.c-p;i.b=b.b-o;i.d=b.d-p;if(Jrb.Gn(a.K,g)>1){if(Jrb.sd(a,g,h,0,j)){i.a+=j.a;i.c+=j.b;if(Jrb.Gn(a.K,g)==2){if(j.a!=0||j.b!=0){i.a+=k.b;i.c-=k.a;}}}else {a.o[g]=new Wrb.CG(i.a,i.c);f=true;}}Jrb.Tk(a.K,c)==386&&Jrb.Jd(e,i);if(Jrb.Tk(a.K,c)==8){if(f){Jrb.ad(a,e,g,h);Jrb.cd(a,i,g,h);}else {Jrb.cd(a,e,g,h);Jrb.ad(a,i,g,h);}}else {Jrb.cd(a,e,g,h);Jrb.cd(a,i,g,h);}}};Jrb.vd=function vd(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;k=a.P;l=a.D;if((a.F&ptb)!=0){a.P=-6;a.D=-8355712;Jrb.Od(a,1);}a.o=tX(Drb.e_,Bsb,52,a.K.q,0,1);for(i=0;i<a.K.r;i++)(Jrb.Tk(a.K,i)==2||Jrb.Tk(a.K,i)==386||Jrb.Tk(a.K,i)==8)&&Jrb.yd(a,i);for(j=0;j<a.K.r;j++)Jrb.Tk(a.K,j)!=2&&Jrb.Tk(a.K,j)!=386&&Jrb.Tk(a.K,j)!=8&&Jrb.yd(a,j);if((a.F&64)==0){for(h=0;h<a.K.r;h++){if(Jrb.Lk(a.K,h)!=0){e=null;Jrb.Lk(a.K,h)==1||Jrb.Lk(a.K,h)==2?(Jrb.Qk(a.K,h)==2||Jrb.Nk(a.K,h)==0||b[Jrb.Nk(a.K,h)][Jrb.Mk(a.K,h)]>1)&&(Jrb.Lk(a.K,h)==1?e=Jrb.Qk(a.K,h)==2?'E':Jrb.ol(a.K,h)?'p':'P':e=Jrb.Qk(a.K,h)==2?'Z':Jrb.ol(a.K,h)?'m':'M'):e='?';if(e!=null){Jrb.at(a,(a.W*2+1)/3|0);Jrb.Od(a,Jrb.ml(a.K,h)?-3:a.K.O==1||(a.F&qtb)!=0?a.P:448);c=Jrb.Ik(a.K,0,h);d=Jrb.Ik(a.K,1,h);n=(Jrb.Qh(a.Q,Jrb.Ak(a.K,c))+Jrb.Qh(a.Q,Jrb.Ak(a.K,d)))/2;o=(Jrb.Rh(a.Q,Jrb.Bk(a.K,c))+Jrb.Rh(a.Q,Jrb.Bk(a.K,d)))/2;f=(Jrb.Qh(a.Q,Jrb.Ak(a.K,c))-Jrb.Qh(a.Q,Jrb.Ak(a.K,d)))/3;g=(Jrb.Rh(a.Q,Jrb.Bk(a.K,c))-Jrb.Rh(a.Q,Jrb.Bk(a.K,d)))/3;Jrb.Bd(a,n+g,o-f,e,true);Jrb.Od(a,a.P);Jrb.at(a,a.W);}}}}if((a.F&4)!=0){Jrb.at(a,(a.W*2+1)/3|0);Jrb.Od(a,384);for(h=0;h<a.K.r;h++){c=Jrb.Ik(a.K,0,h);d=Jrb.Ik(a.K,1,h);m=Jrb.Do(a.K,h)?'d':Jrb.zo(a.K,h)?'a':'';n=(Jrb.Qh(a.Q,Jrb.Ak(a.K,c))+Jrb.Qh(a.Q,Jrb.Ak(a.K,d)))/2;o=(Jrb.Rh(a.Q,Jrb.Bk(a.K,c))+Jrb.Rh(a.Q,Jrb.Bk(a.K,d)))/2;Jrb.Bd(a,n,o,m+(''+h),true);}Jrb.Od(a,a.P);Jrb.at(a,a.W);}if((a.F&ptb)!=0){a.P=k;a.D=l;}};Jrb.wd=function wd(a){var b,c;for(c=new Krb.ueb(a.T);c.a<c.c.a.length;){b=Krb.teb(c);Jrb.Od(a,b.a);Jrb.bd(a,b.b,b.c);}Jrb.Od(a,a.P);};Jrb.xd=function xd(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$;a.J||Jrb.Ys(a,b,(Jrb.sk(a.K,b),Jrb.Qh(a.Q,Jrb.Ak(a.K,b))),Jrb.Rh(a.Q,Jrb.Bk(a.K,b)));M=null;if(Jrb.mk(a.K,b)!=0){U=$wnd.Math.abs(Jrb.mk(a.K,b))==1?'':''+$wnd.Math.abs(Jrb.mk(a.K,b));M=Jrb.mk(a.K,b)<0?U+'-':U+'+';}D=null;N=Jrb.yk(a.K,b);if(o3(N,0)!=0){E3(j3(N,rtb),0)&&(D=D==null?'*':D+','+'*');E3(j3(N,stb),0)&&(D=D==null?'!*':D+','+'!*');E3(j3(N,ttb),0)?D=D==null?'ha':D+','+'ha':E3(j3(N,2),0)?D=D==null?'a':D+','+'a':E3(j3(N,4),0)&&(D=D==null?'!a':D+','+'!a');E3(j3(N,qtb),0)&&(D=D==null?'s':D+','+'s');if(E3(j3(N,utb),0)){w=j3(N,utb);o3(w,1792)==0?D=D==null?'h0':D+','+'h0':o3(w,1664)==0?D=D==null?'h1':D+','+'h1':o3(w,1408)==0?D=D==null?'h2':D+','+'h2':o3(w,128)==0?D=D==null?'h>0':D+','+'h>0':o3(w,384)==0?D=D==null?'h>1':D+','+'h>1':o3(w,896)==0?D=D==null?'h>2':D+','+'h>2':o3(w,vtb)==0?D=D==null?'h<3':D+','+'h<3':o3(w,1536)==0?D=D==null?'h<2':D+','+'h<2':o3(w,1152)==0&&(D=D==null?'h1-2':D+','+'h1-2');}if(E3(j3(N,wtb),0)){i=j3(N,wtb);o3(i,xtb)==0?D=D==null?'c0':D+','+'c0':o3(i,ytb)==0?D=D==null?'c+':D+','+'c+':o3(i,ztb)==0&&(D=D==null?'c-':D+','+'c-');}if(E3(j3(N,Atb),0)){L=j3(N,Atb);o3(L,98304)==0?D=D==null?'pi0':D+','+'pi0':o3(L,81920)==0?D=D==null?'pi1':D+','+'pi1':o3(L,49152)==0?D=D==null?'pi2':D+','+'pi2':o3(L,ptb)==0&&(D=D==null?'pi>0':D+','+'pi>0');}if(E3(j3(N,Btb),0)){K=j3(N,Btb);o3(K,3801088)==0?D=D==null?'n1':D+','+'n1':o3(K,Ctb)==0?D=D==null?'n2':D+','+'n2':o3(K,Dtb)==0?D=D==null?'n3':D+','+'n3':o3(K,3145728)==0?D=D==null?'n<3':D+','+'n<3':o3(K,Etb)==0?D=D==null?'n<4':D+','+'n<4':o3(K,Ftb)==0?D=D==null?'n>1':D+','+'n>1':o3(K,Gtb)==0?D=D==null?'n>2':D+','+'n>2':o3(K,Htb)==0?D=D==null?'n>3':D+','+'n>3':o3(K,3276800)==0?D=D==null?'n1-2':D+','+'n1-2':o3(K,2228224)==0?D=D==null?'n1-3':D+','+'n1-3':o3(K,2490368)==0&&(D=D==null?'n2-3':D+','+'n2-3');}if(E3(j3(N,Itb),0)){o=j3(N,Itb);o3(o,Jtb)==0?D=D==null?'e0':D+','+'e0':o3(o,Ktb)==0?D=D==null?'e1':D+','+'e1':o3(o,Ltb)==0?D=D==null?'e2':D+','+'e2':o3(o,Mtb)==0?D=D==null?'e3':D+','+'e3':o3(o,15393162788864)==0?D=D==null?'e<2':D+','+'e<2':o3(o,13194139533312)==0?D=D==null?'e<3':D+','+'e<3':o3(o,Ntb)==0?D=D==null?'e<4':D+','+'e<4':o3(o,Otb)==0?D=D==null?'e>0':D+','+'e>0':o3(o,Ptb)==0?D=D==null?'e>1':D+','+'e>1':o3(o,Qtb)==0?D=D==null?'e>2':D+','+'e>2':o3(o,Rtb)==0?D=D==null?'e>3':D+','+'e>3':o3(o,13743895347200)==0?D=D==null?'e1-2':D+','+'e1-2':o3(o,9345848836096)==0?D=D==null?'e1-3':D+','+'e1-3':o3(o,10445360463872)==0&&(D=D==null?'e2-3':D+','+'e2-3');}if(E3(j3(N,120),0)){Q=j3(N,120);o3(Q,112)==0?D=D==null?'!r':D+','+'!r':o3(Q,8)==0?D=D==null?'r':D+','+'r':o3(Q,96)==0?D=D==null?'rb<3':D+','+'rb<3':o3(Q,104)==0?D=D==null?'rb2':D+','+'rb2':o3(Q,88)==0?D=D==null?'rb3':D+','+'rb3':o3(Q,56)==0&&(D=D==null?'rb4':D+','+'rb4');}E3(j3(N,Stb),0)&&(D=D==null?'r'+O3(I3(j3(N,Stb),22)):D+','+('r'+O3(I3(j3(N,Stb),22))));E3(j3(N,Usb),0)&&(D=Jrb.Yc(D,Jrb.$c(N)));E3(j3(N,Ttb),0)&&(D=D==null?'f':D+','+'f');}Jrb.wk(a.K,b)!=0&&(D=Jrb.Yc(D,''+Jrb.wk(a.K,b)));T=0;if(Jrb.zk(a.K,b)!=0){switch(Jrb.zk(a.K,b)){case 16:M=M==null?'|':M+','+'|';break;case 32:T=1;break;case 48:T=2;}}l=null;if((a.F&64)==0){if(Jrb.fl(a.K,b))l='?';else if(Jrb.lk(a.K,b)!=0){if(Jrb.rk(a.K,b)==0||c==null||c[Jrb.rk(a.K,b)][Jrb.qk(a.K,b)]>1){if(Jrb.Gn(a.K,b)==2){switch(Jrb.lk(a.K,b)){case 2:l=Jrb.hl(a.K,b)?'p':'P';break;case 1:l=Jrb.hl(a.K,b)?'m':'M';break;default:l='*';}}else {switch(Jrb.lk(a.K,b)){case 1:l=Jrb.hl(a.K,b)?'r':'R';break;case 2:l=Jrb.hl(a.K,b)?'s':'S';break;default:l='*';}}}}}(a.F&768)!=0&&(l=Jrb.Yc(l,''+Jrb.zu(a.K,b)));I=null;(a.F&16)!=0&&Jrb.vk(a.K,b)!=0&&(I=''+Jrb.vk(a.K,b));q=null;if(Jrb.no(a.K,b)!=-1){p=Jrb.kd(a,b);p!=-1&&(q=p==0?'abs':((p&255)==1?'&':'or')+(1+(p>>8)));}A=0;(a.F&Utb)==0&&(a.K.K?E3(j3(Jrb.yk(a.K,b),Vtb),0)&&(A=Jrb.Wn(a.K,b)):(Jrb.Dk(a.K,b)!=6||Jrb.wk(a.K,b)!=0||!a.q[b]||Jrb.zk(a.K,b)!=0)&&(A=Jrb.Wn(a.K,b)));H=false;f=Jrb.ok(a.K,b);if(f!=null&&Erb.k7(Hrb.rnb(f).substr(0,1),']')){D=Jrb.Yc((Hrb.mnb(1,Hrb.rnb(f).length+1),Hrb.rnb(f).substr(1)),D);f=null;H=true;}if(f!=null){A=0;}else if(Jrb.tk(a.K,b)!=null){e=E3(j3(Jrb.yk(a.K,b),1),0)?'[!':'[';f=e+Jrb.uk(a.K,b)+']';Hrb.rnb(f).length>5&&(f=e+Jrb.tk(a.K,b).length+']');E3(j3(Jrb.yk(a.K,b),Vtb),0)&&(A=-1);}else if(E3(j3(Jrb.yk(a.K,b),1),0)){f='?';E3(j3(Jrb.yk(a.K,b),Vtb),0)&&(A=-1);}else (Jrb.Dk(a.K,b)!=6||M!=null||D!=null||A>0||!a.q[b])&&(f=Jrb.sk(a.K,b));G=0;!Jrb.xl(a.K,b)&E3(j3(Jrb.yk(a.K,b),jtb),0)&&Jrb.Od(a,-8);if(f!=null){G=(O=(V=dsb.m4(a.e,f),new esb.s4(V)).b,O);Jrb.Bd(a,Jrb.Qh(a.Q,Jrb.Ak(a.K,b)),Jrb.Rh(a.Q,Jrb.Bk(a.K,b)),f,true);a.s[b]=true;}else Jrb.rd(a,b)&&Jrb.Ad(a,Jrb.Qh(a.Q,Jrb.Ak(a.K,b)),Jrb.Rh(a.Q,Jrb.Bk(a.K,b)),b);if(M!=null){Jrb.at(a,(a.W*2+1)/3|0);X=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))+((G+(O=(V=dsb.m4(a.e,M),new esb.s4(V)).b,O))/2+1);Z=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))-((a.k*4-4)/8|0);Jrb.Bd(a,X,Z,M,true);Jrb.at(a,a.W);}(a.F&2)!=0&&(D=''+b);if(D!=null){H?Jrb.at(a,(a.W*5+1)/6|0):Jrb.at(a,(a.W*2+1)/3|0);X=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))-(G+(O=(V=dsb.m4(a.e,D),new esb.s4(V)).b,O))/2;Z=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))-((a.k*4-4)/8|0);Jrb.Bd(a,X,Z,D,true);Jrb.at(a,a.W);}if(l!=null){Jrb.at(a,(a.W*2+1)/3|0);X=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))-(G+(O=(V=dsb.m4(a.e,l),new esb.s4(V)).b,O))/2;Z=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))+((a.k*4+4)/8|0);S=a.B;a.K.O!=1&&(a.F&qtb)==0&&Jrb.Od(a,448);Jrb.Bd(a,X,Z,l,false);Jrb.Od(a,S);Jrb.at(a,a.W);}if(I!=null){Jrb.at(a,(a.W*2+1)/3|0);X=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))+((G+(O=(V=dsb.m4(a.e,I),new esb.s4(V)).b,O))/2+1);Z=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))+((a.k*4+4)/8|0);S=a.B;Jrb.Od(a,Jrb.jl(a.K,b)?384:448);Jrb.Bd(a,X,Z,I,true);Jrb.Od(a,S);Jrb.at(a,a.W);}if(q!=null){d=Jrb.Fd(a,b);Jrb.at(a,(a.W*2+1)/3|0);X=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))+0.7*a.k*$wnd.Math.sin(d);Z=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))+0.7*a.k*$wnd.Math.cos(d);S=a.B;!a.J&&a.K.O!=1&&Jrb.Od(a,Jrb.jd(a,b));Jrb.Bd(a,X,Z,q,false);Jrb.Od(a,S);Jrb.at(a,a.W);}if(A==0&&T==0){a.B==-8&&Jrb.Od(a,-9);return;}u=tX(Drb.LY,Osb,6,4,15,1);for(C=0;C<Jrb.sn(a.K,b);C++){h=Jrb.Hn(a.K,b,C);for(F=0;F<2;F++){if(Jrb.Ik(a.K,F,h)==b){R=Jrb.Hk(a.K,Jrb.Ik(a.K,F,h),Jrb.Ik(a.K,1-F,h));if(R<Wtb){u[0]-=R+otb;u[3]+=R+ntb;}else if(R<0){u[2]+=R+otb;u[3]-=R;}else if(R<otb){u[1]+=R;u[2]+=otb-R;}else {u[0]+=R-otb;u[1]+=ntb-R;}}}}Jrb.Gn(a.K,b)==0?Jrb.ql(a.K,b)?u[3]-=0.2:u[1]-=0.2:u[1]-=0.1;(M!=null||I!=null)&&(u[1]+=10);(D!=null||l!=null)&&(u[3]+=10);s='';if(A!=0){v=(P=(W=dsb.m4(a.e,'H'),new esb.s4(W)).b,P);t=0;r=a.k;if(A==-1){s='n';Jrb.at(a,(a.W*2+1)/3|0);t=(O=(V=dsb.m4(a.e,s),new esb.s4(V)).b,O);}else if(A>1){s=''+A;Jrb.at(a,(a.W*2+1)/3|0);t=(O=(V=dsb.m4(a.e,s),new esb.s4(V)).b,O);}if(u[1]<0.6||u[3]<0.6){k=Jrb.Rh(a.Q,Jrb.Bk(a.K,b));if(u[1]<=u[3]){u[1]+=10;j=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))+(G+v)/2;}else {u[3]+=10;j=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))-(G+v)/2-t;}}else {j=Jrb.Qh(a.Q,Jrb.Ak(a.K,b));if(u[0]<u[2]){u[0]+=10;k=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))-r;}else {u[2]+=10;k=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))+r;}}if(t>0){X=j+(v+t)/2;Z=k+((a.k*4+4)/8|0);Jrb.Bd(a,X,Z,s,true);Jrb.at(a,a.W);}Jrb.Bd(a,j,k,'H',true);}g=0;if(T!=0){J=50;m=0;for(B=0;B<4;B++){n=B>1?B-2:B+2;if(u[B]<J){g=B;J=u[B];m=u[n];}else if(u[B]==J){if(u[n]>m){g=B;m=u[n];}}}switch(g){case 0:j=Jrb.Qh(a.Q,Jrb.Ak(a.K,b));k=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))-a.U-G/2;break;case 1:j=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))+a.U+G/2;k=Jrb.Rh(a.Q,Jrb.Bk(a.K,b));break;case 2:j=Jrb.Qh(a.Q,Jrb.Ak(a.K,b));k=Jrb.Rh(a.Q,Jrb.Bk(a.K,b))+a.U+G/2;break;default:j=Jrb.Qh(a.Q,Jrb.Ak(a.K,b))-a.U-G/2;k=Jrb.Rh(a.Q,Jrb.Bk(a.K,b));}if(T==1){Krb.Ei(a.Z,new Wrb.LG(j-a.U,k-a.U,2*a.U,2*a.U));a.J||Krb.Ei(a.T,new Jrb.Vd(j,k,Jrb.pd(a,b)?-3:a.p[b]));}else {switch(g){case 2:case 0:Y=2*a.U;$=0;j-=a.U;break;case 1:Y=0;$=2*a.U;k-=a.U;break;default:Y=0;$=2*a.U;k-=a.U;}Krb.Ei(a.Z,new Wrb.LG(j-a.U,k-a.U,2*a.U,2*a.U));a.J||Krb.Ei(a.T,new Jrb.Vd(j,k,Jrb.pd(a,b)?-3:a.p[b]));Krb.Ei(a.Z,new Wrb.LG(j+Y-a.U,k+$-a.U,2*a.U,2*a.U));a.J||Krb.Ei(a.T,new Jrb.Vd(j+Y,k+$,Jrb.pd(a,b)?-3:a.p[b]));}}a.B==-8&&Jrb.Od(a,-9);};Jrb.yd=function yd(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w;u=new Jrb.Wd();c=new Jrb.Wd();g=new Jrb.Wd();q=new Wrb.BG();p=new Wrb.BG();e=Jrb.Ik(a.K,0,b);f=Jrb.Ik(a.K,1,b);Jrb.Zs(a,b,Jrb.Qh(a.Q,Jrb.Ak(a.K,e)),Jrb.Rh(a.Q,Jrb.Bk(a.K,e)),Jrb.Qh(a.Q,Jrb.Ak(a.K,f)),Jrb.Rh(a.Q,Jrb.Bk(a.K,f)));!Jrb.xl(a.K,e)&&!Jrb.xl(a.K,f)&&E3(j3(G3(Jrb.yk(a.K,e),Jrb.yk(a.K,f)),jtb),0)&&Jrb.Od(a,-8);if(!a.o[e]){u.a=Jrb.Qh(a.Q,Jrb.Ak(a.K,e));u.c=Jrb.Rh(a.Q,Jrb.Bk(a.K,e));}else {u.a=a.o[e].a;u.c=a.o[e].b;}if(!a.o[f]){u.b=Jrb.Qh(a.Q,Jrb.Ak(a.K,f));u.d=Jrb.Rh(a.Q,Jrb.Bk(a.K,f));}else {u.b=a.o[f].a;u.d=a.o[f].b;}if((Jrb.Sk(a.K,b)&Xtb)!=0){Jrb.Kd(a,u)&&Jrb.Us(a,u);Jrb.Od(a,-9);return;}h=Jrb.Tk(a.K,b)==8?0:Jrb.Tk(a.K,b)==16?1:Jrb.Qk(a.K,b);switch(h){case 1:i=Jrb.Tk(a.K,b);if((a.F&128)!=0&&(i==257||i==129)){t=Jrb.Ik(a.K,0,b);n=Jrb.rk(a.K,t);if(n!=0){m=Jrb.qk(a.K,t);l=0;for(d=0;d<a.K.f;d++)Jrb.rk(a.K,d)==n&&Jrb.qk(a.K,d)==m&&++l;l==1&&(i=1);}}switch(i){case 1:Jrb.Kd(a,u)&&Jrb.cd(a,u,e,f);break;case 257:Jrb.Hd(a,u,e,f);break;case 129:v=u.b-u.a;w=u.d-u.c;if(Jrb.ml(a.K,Jrb.Dn(a.K,e,f))){j=-3;k=-3;}else {j=a.p[e];k=Jrb.jd(a,e);j==Jrb.nk(a.K,e)&&(j=k);}for(o=2;o<17;o+=2){c.a=u.a+o*v/17-o*w/128;c.c=u.c+o*w/17+o*v/128;c.b=u.a+o*v/17+o*w/128;c.d=u.c+o*w/17-o*v/128;if(Jrb.Kd(a,c)){Jrb.Od(a,o<9?j:k);r=atb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c.a),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c.d),gsb.xab)))+btb+ftb+a.d+btb+gtb+a.j+htb;Jrb.dt(a,r);Jrb.Od(a,a.P);}}break;case 16:Jrb.Kd(a,u)&&Jrb.ed(a,u,e,f);}break;case 0:case 2:if((a.s[e]||Jrb.wn(a.K,e)==2)&&(a.s[f]||Jrb.wn(a.K,f)==2)&&!Jrb.Mo(a.K,b)&&h==2){if(!Jrb.Kd(a,u))break;Jrb.td(a,u.b-u.a,u.d-u.c,q);v=q.a/2;w=q.b/2;c.a=u.a+v;c.c=u.c+w;c.b=u.b+v;c.d=u.d+w;g.a=u.a-v;g.c=u.c-w;g.b=u.b-v;g.d=u.d-w;Jrb.Tk(a.K,b)==386&&Jrb.Jd(c,g);Jrb.cd(a,c,e,f);h==2?Jrb.cd(a,g,e,f):Jrb.ad(a,g,e,f);}else if((a.s[f]||Jrb.wn(a.K,f)==2)&&h==2){Jrb.ud(a,u,b,false);}else if((a.s[e]||Jrb.wn(a.K,e)==2)&&h==2){Jrb.ud(a,u,b,true);}else {s=Jrb.io(a.K,b);s==0&&(s=1);c.a=u.a;c.c=u.c;c.b=u.b;c.d=u.d;Jrb.td(a,u.b-u.a,u.d-u.c,q);if(s>0){g.a=u.a+q.a;g.c=u.c+q.b;g.b=u.b+q.a;g.d=u.d+q.b;if(Jrb.sd(a,e,f,1,p)||Jrb.Gn(a.K,e)>1){g.a+=p.a+q.b;g.c+=p.b-q.a;}if(Jrb.sd(a,f,e,-1,p)||Jrb.Gn(a.K,f)>1){g.b+=p.a-q.b;g.d+=p.b+q.a;}}else {g.a=u.a-q.a;g.c=u.c-q.b;g.b=u.b-q.a;g.d=u.d-q.b;if(Jrb.sd(a,e,f,-1,p)||Jrb.Gn(a.K,e)>1){g.a+=p.a+q.b;g.c+=p.b-q.a;}if(Jrb.sd(a,f,e,1,p)||Jrb.Gn(a.K,f)>1){g.b+=p.a-q.b;g.d+=p.b+q.a;}}Jrb.Tk(a.K,b)==386&&Jrb.Jd(c,g);Jrb.Kd(a,c)&&Jrb.cd(a,c,e,f);h==2?Jrb.Kd(a,g)&&Jrb.cd(a,g,e,f):Jrb.Kd(a,g)&&Jrb.ad(a,g,e,f);}break;case 3:if(Jrb.Kd(a,u)){Jrb.cd(a,u,e,f);Jrb.td(a,u.b-u.a,u.d-u.c,q);Jrb.dd(a,u,e,f,q.a,q.b,c);Jrb.dd(a,u,e,f,-q.a,-q.b,c);}break;case 4:if(Jrb.Kd(a,u)){Jrb.td(a,u.b-u.a,u.d-u.c,q);Jrb.dd(a,u,e,f,1.5*q.a,1.5*q.b,c);Jrb.dd(a,u,e,f,0.5*q.a,0.5*q.b,c);Jrb.dd(a,u,e,f,-0.5*q.a,-0.5*q.b,c);Jrb.dd(a,u,e,f,-1.5*q.a,-1.5*q.b,c);}break;case 5:if(Jrb.Kd(a,u)){Jrb.cd(a,u,e,f);Jrb.td(a,u.b-u.a,u.d-u.c,q);Jrb.dd(a,u,e,f,2*q.a,2*q.b,c);Jrb.dd(a,u,e,f,q.a,q.b,c);Jrb.dd(a,u,e,f,-q.a,-q.b,c);Jrb.dd(a,u,e,f,-2*q.a,-2*q.b,c);}}a.B==-8&&Jrb.Od(a,-9);};Jrb.zd=function zd(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r;o=false;for(d=0;d<a.K.g;d++){j=null;if(Jrb.ll(a.K,d)){l=Jrb.Kk(a.K,d);k=Jrb.Jk(a.K,d);j=l==k?'['+l+']':'['+l+':'+k+']';}else (Jrb.Sk(a.K,d)&Ytb)!=0?j=(Jrb.Sk(a.K,d)&Ytb)==Etb?'a':(Jrb.Sk(a.K,d)&384)==256?'r!a':'!a':(Jrb.Sk(a.K,d)&384)!=0&&(j=(Jrb.Sk(a.K,d)&384)==256?'r':'!r');n=(Jrb.Sk(a.K,d)&Gtb)>>17;n!=0&&(j=(j==null?'':j)+n);if(j!=null){b=Jrb.Ik(a.K,0,d);c=Jrb.Ik(a.K,1,d);if(!o){Jrb.at(a,(a.W*2+1)/3|0);o=true;}q=(Jrb.Qh(a.Q,Jrb.Ak(a.K,b))+Jrb.Qh(a.Q,Jrb.Ak(a.K,c)))/2;r=(Jrb.Rh(a.Q,Jrb.Bk(a.K,b))+Jrb.Rh(a.Q,Jrb.Bk(a.K,c)))/2;f=Jrb.Qh(a.Q,Jrb.Ak(a.K,c))-Jrb.Qh(a.Q,Jrb.Ak(a.K,b));g=Jrb.Rh(a.Q,Jrb.Bk(a.K,c))-Jrb.Rh(a.Q,Jrb.Bk(a.K,b));e=$wnd.Math.sqrt(f*f+g*g);i=(m=(p=dsb.m4(a.e,j),new esb.s4(p)).b,0.6*m);h=0.55*a.k;e!=0&&(f>0?Jrb.Bd(a,q+i*g/e,r-h*f/e,j,true):Jrb.Bd(a,q-i*g/e,r+h*f/e,j,true));}}o&&Jrb.at(a,a.W);};Jrb.Ad=function Ad(a,b,c,d){Krb.Ei(a.Z,new Wrb.LG(b-a.U,c-a.U,2*a.U,2*a.U));a.J||Krb.Ei(a.T,new Jrb.Vd(b,c,Jrb.pd(a,d)?-3:a.p[d]));};Jrb.Bd=function Bd(a,b,c,d,e){var f,g,h,i,j;if(e){g=(f=(h=dsb.m4(a.e,d),new esb.s4(h)).b,f);i=g/2+(a.k/8|0);j=a.k/2|0;(d=='+'||d=='-')&&(j=j*2/3);Krb.Ei(a.Z,new Wrb.LG(b-i,c-j,2*i,2*j));}a.J||Jrb.Ws(a,d,b,c);};Jrb.Cd=function Cd(a){var b;b=a.a;a.a=a.b;a.b=b;b=a.c;a.c=a.d;a.d=b;};Jrb.Dd=function Dd(a,b,c){var d;d=b==0?mtb+a[0]-a[a.length-1]:a[b]-a[b-1];c>-2.0943951023931953&&c<Ztb?d-=2*$wnd.Math.cos(c+$tb):d-=0.5*$wnd.Math.cos(c+$tb);return d;};Jrb.Ed=function Ed(a){var b;b=new Wrb.KG();if(a.a<=a.b){b.c=a.a;b.b=a.b-a.a;}else {b.c=a.b;b.b=a.a-a.b;}if(a.c<=a.d){b.d=a.c;b.a=a.d-a.c;}else {b.d=a.d;b.a=a.c-a.d;}return b;};Jrb.Fd=function Fd(a,b){var c,d,e,f,g,h,i;c=tX(Drb.LY,Osb,6,Jrb.rn(a.K,b),15,1);for(e=0;e<Jrb.rn(a.K,b);e++)c[e]=Jrb.Hk(a.K,b,Jrb.Fn(a.K,b,e));Hrb.Nmb(c,Hrb.rnb(a4(Krb.$eb.prototype.pc,Krb.$eb,[])));f=Jrb.Gd(c,0);g=Jrb.Dd(c,0,f);for(d=1;d<c.length;d++){h=Jrb.Gd(c,d);i=Jrb.Dd(c,d,h);if(g<i){g=i;f=h;}}return f;};Jrb.Gd=function Gd(a,b){var c;if(b>0)return (a[b]+a[b-1])/2;c=ntb+(a[0]+a[a.length-1])/2;return c>ntb?c-mtb:c;};Jrb.Hd=function Hd(a,b,c,d){var e,f,g,h;h=new Jrb.Wd();if(b.a==b.b&&b.c==b.d)return;h.a=b.a;h.c=b.c;h.b=b.b;h.d=b.d;g=Jrb.Ed(h);for(e=0;e<a.Z.a.length;e++){f=Krb.Ji(a.Z,e);if(f.c>g.c+g.b||f.d>g.d+g.a||g.c>f.c+f.b||g.d>f.d+f.a)continue;if(Jrb.Id(a,h.a,h.c,e)){if(Jrb.Id(a,h.b,h.d,e))return;Jrb.Ld(a,h,0,e);Jrb.Hd(a,h,c,d);return;}if(Jrb.Id(a,h.b,h.d,e)){Jrb.Ld(a,h,1,e);Jrb.Hd(a,h,c,d);return;}}Jrb.fd(a,h,c,d);};Jrb.Id=function Id(a,b,c,d){var e;if((a.F&1)!=0)return false;e=Krb.Ji(a.Z,d);return b>e.c&&b<e.c+e.b&&c>e.d&&c<e.d+e.a;};Jrb.Jd=function Jd(a,b){var c;c=a.b;a.b=b.b;b.b=c;c=a.d;a.d=b.d;b.d=c;};Jrb.Kd=function Kd(a,b){var c,d,e,f,g,h;if(b.a==b.b&&b.c==b.d){for(g=new Krb.ueb(a.Z);g.a<g.c.a.length;){f=Krb.teb(g);if(Wrb.GG(f,b.a,b.c))return false;}return true;}h=Jrb.Ed(b);c=false;if(b.a>b.b){Jrb.Cd(b);c=true;}for(d=0;d<a.Z.a.length;d++){f=Krb.Ji(a.Z,d);if(f.c>h.c+h.b||f.d>h.d+h.a||h.c>f.c+f.b||h.d>f.d+f.a)continue;if(Jrb.Id(a,b.a,b.c,d)){if(Jrb.Id(a,b.b,b.d,d)){c&&Jrb.Cd(b);return false;}Jrb.Ld(a,b,0,d);e=Jrb.Kd(a,b);c&&Jrb.Cd(b);return e;}if(Jrb.Id(a,b.b,b.d,d)){Jrb.Ld(a,b,1,d);e=Jrb.Kd(a,b);c&&Jrb.Cd(b);return e;}}c&&Jrb.Cd(b);return true;};Jrb.Ld=function Ld(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o;if(c==0){l=b.a;n=b.c;m=b.b;o=b.d;}else {l=b.b;n=b.d;m=b.a;o=b.c;}k=Krb.Ji(a.Z,d);i=m>l?k.c+k.b:k.c;j=o>n?k.d+k.a:k.d;e=m-l;f=o-n;if($wnd.Math.abs(e)>$wnd.Math.abs(f)){if(n==o){g=i;h=n;}else {g=l+e*(j-n)/f;if(m>l==i>g){h=j;}else {g=i;h=n+f*(i-l)/e;}}}else {if(l==m){g=l;h=j;}else {h=n+f*(i-l)/e;if(o>n==j>h){g=i;}else {g=l+e*(j-n)/f;h=j;}}}if(c==0){b.a=g;b.c=h;}else {b.b=g;b.d=h;}};Jrb.Md=function Md(a){var b,c,d,e,f,g;if(a.K.q==0)return;Jrb.ou(a.K,(a.F&256)!=0?63:(a.F&512)!=0?95:31);Jrb.Zc(a);d=Jrb.tu(a.K);e=false;a.p=tX(Drb.NY,Qsb,6,a.K.q,15,1);for(b=0;b<a.K.q;b++){a.p[b]=Jrb.nk(a.K,b);a.p[b]!=0&&(e=true);Jrb.xl(a.K,b)&&(a.p[b]=128);Jrb.al(a.K,b)&&(a.F&Vtb)==0&&(a.p[b]=256);}Jrb.Od(a,-10);Jrb.md(a);Jrb.ld(a);Jrb.nd(a);Jrb.Xc(a);Jrb.at(a,a.W);Jrb.$s(a,a.X);Jrb.Od(a,a.P);Jrb.qd(a);Hrb.Mmb(a.T.a,0);Hrb.Mmb(a.Z.a,0);if((a.F&1)!=0){Jrb.vd(a,d);Jrb.wd(a);Jrb.zd(a);}for(f=0;f<a.K.q;f++){if(Jrb.pd(a,f)){Jrb.Od(a,-3);Jrb.xd(a,f,d);Jrb.Od(a,a.P);}else if(a.p[f]!=0){Jrb.Od(a,a.p[f]);Jrb.xd(a,f,d);Jrb.Od(a,a.P);}else if(!e&&a.K.O!=1&&Jrb.Dk(a.K,f)!=1&&Jrb.Dk(a.K,f)!=6&&(a.F&vtb)==0&&Jrb.tk(a.K,f)==null&&Jrb.Dk(a.K,f)<Jrb.Uc.length&&Jrb.Uc[Jrb.Dk(a.K,f)]!=0){g=Jrb.Uc[Jrb.Dk(a.K,f)];Jrb.Pd(a,(c=a.L!=0?a.L:a.C!=0?a.C:-1,Zrb._U(g,c)));Jrb.xd(a,f,d);Jrb.Od(a,a.P);}else {Jrb.xd(a,f,d);}}if((a.F&1)==0){Jrb.wd(a);Jrb.zd(a);Jrb.vd(a,d);}};Jrb.Nd=function Nd(a,b,c,d){var e;e=c/2;switch(d&_tb){case _tb:if(b){a.w.a=b.c+b.b/2;a.w.b=b.d+b.a-e;break;}case 0:a.w.a=a.v.c+a.v.b/2;a.w.b=a.v.d+a.v.a+e;!!b&&a.w.b>b.d+b.a-e&&(a.w.b=b.d+b.a-e);break;case aub:if(b){a.w.a=b.c+b.b/2;a.w.b=b.d+e;break;}case bub:a.w.a=a.v.c+a.v.b/2;a.w.b=a.v.d-e;!!b&&a.w.b<b.d+e&&(a.w.b=b.d+e);}};Jrb.Od=function Od(a,b){if(a.J)return;if(b==-10){a.B=-999;b=a.P;}b!=-2&&b!=-7&&a.M!=0&&(b=-4);if(b==a.B)return;if(a.B==-8&&b!=-9)return;b==-8&&(a.N=a.B);b==-9&&(b=a.N);a.B=b;switch(b){case 0:Jrb._s(a,a.D==0?cub:a.D);break;case  -6:Jrb._s(a,a.D);break;case  -4:Jrb._s(a,a.M);break;case  -2:Jrb._s(a,a.t);break;case  -3:Jrb._s(a,a.u);break;case  -7:Jrb._s(a,a.G);break;case  -8:Jrb._s(a,a.H);break;case  -5:Jrb._s(a,a.O);break;case 64:a.d='rgb(32,96,255)';break;case 128:a.d='rgb(255,0,0)';break;case 256:a.d='rgb(192,0,255)';break;case 192:a.d='rgb(0,255,0)';break;case 320:a.d='rgb(255,160,0)';break;case 384:a.d='rgb(0,128,0)';break;case 448:a.d='rgb(160,0,0)';break;case 1:a.d='rgb(128,128,128)';break;default:a.d='rgb(0,0,0)';}};Jrb.Pd=function Pd(a,b){if(a.M!=0){if(a.B!=-4){a.B=-4;Jrb._s(a,a.M);}return;}a.B=-5;a.O=b;a.d='rgb('+((b&dub)>>16)+','+((b&Ssb)>>8)+','+(b&255)+')';};Jrb.Qd=function Qd(a){var b,c,d,e,f;e=Jrb.Qh(a.Q,Jrb.Ak(a.K,0));c=Jrb.Qh(a.Q,Jrb.Ak(a.K,0));f=Jrb.Rh(a.Q,Jrb.Bk(a.K,0));d=Jrb.Rh(a.Q,Jrb.Bk(a.K,0));for(b=0;b<a.K.q;b++){e>Jrb.Qh(a.Q,Jrb.Ak(a.K,b))&&(e=Jrb.Qh(a.Q,Jrb.Ak(a.K,b)));c<Jrb.Qh(a.Q,Jrb.Ak(a.K,b))&&(c=Jrb.Qh(a.Q,Jrb.Ak(a.K,b)));f>Jrb.Rh(a.Q,Jrb.Bk(a.K,b))&&(f=Jrb.Rh(a.Q,Jrb.Bk(a.K,b)));d<Jrb.Rh(a.Q,Jrb.Bk(a.K,b))&&(d=Jrb.Rh(a.Q,Jrb.Bk(a.K,b)));}return new Wrb.LG(e,f,c-e,d-f);};Jrb.Rd=function Rd(a,b){var c,d;if(a.K.q==0)return null;a.v=Jrb.Qd(a);c=a.Q.c*Jrb.Ek(a.K);d=new Jrb.Th(a.v,b,c);if(d.c==1&&d.a==0&&d.b==0){d=null;}else {Jrb.Mh(d,a.Q);Jrb.Oh(d,a.v);}Jrb.Nd(a,b,c,Fsb);return d;};Jrb.Sd=function Sd(a){var b;b=a.L!=0?a.L:a.C!=0?a.C:-1;a.t=Zrb.aV(b);a.u=Zrb._U(eub,b);a.G=-24321;a.H=-6291392;};Jrb.Td=function Td(a,b){var c,d,e,f;if(a.K.q==0)return null;e=Jrb.bt(a,b);Jrb.ou(a.K,(a.F&256)!=0?63:(a.F&512)!=0?95:31);Jrb.qd(a);Hrb.Mmb(a.T.a,0);Hrb.Mmb(a.Z.a,0);Jrb.Zc(a);Jrb.at(a,a.W);a.J=true;for(d=0;d<a.K.q;d++)Jrb.xd(a,d,null);a.J=false;c=a.Q.c*Jrb.Ek(a.K);Jrb.gd(a,c);Jrb.Nd(a,b,c,Fsb);if(Wrb.HG(b,a.v))return e;f=new Jrb.Th(a.v,b,c);Jrb.Mh(f,a.Q);Jrb.Oh(f,a.v);Jrb.Nh(f,a.w);if(!e)return f;Jrb.Mh(f,e);return e;};Jrb.Ud=function Ud(a,b){Jrb.Wc(this);this.K=a;this.F=b;Jrb.od(this);};Z3(263,1,{});_.r=0;_.t=0;_.u=0;_.A=0;_.B=0;_.C=0;_.D=0;_.F=0;_.G=0;_.H=0;_.I=0;_.J=false;_.L=0;_.M=0;_.N=0;_.O=0;_.P=0;_.R=0;_.S=0;_.U=0;_.V=0;_.W=0;_.X=0;_.Y=0;Drb.SY=B5(263);Jrb.Vd=function Vd(a,b,c){this.b=a;this.c=b;this.a=c;};Z3(119,1,{119:1},Jrb.Vd);_.a=0;_.b=0;_.c=0;Drb.QY=B5(119);Jrb.Wd=function Wd(){};Z3(62,1,{},Jrb.Wd);_.a=0;_.b=0;_.c=0;_.d=0;Drb.RY=B5(62);Z3(169,1,{169:1});_.mb=function Xd(){var a;a=new Erb.I7();Erb.H7(a,fub+this.pb()+'"'+this.ob()+gub);return a.a;};_.d=false;Drb.TY=B5(169);Jrb.Yd=function Yd(a,b,c){if(a.f[b])return 0;if(Jrb.Dk(a.k,b)==7){if(Jrb.rn(a.k,b)==3)return 6;else if(Jrb.Gn(a.k,b)==2)return a.a?0:4;}else if(Jrb.Dk(a.k,b)==8){return 10;}else if(Jrb.Dk(a.k,b)==15||Jrb.Dk(a.k,b)==33){if(Jrb.Gn(a.k,b)==3)return 8;}else if(Jrb.Dk(a.k,b)==16||Jrb.Dk(a.k,b)==34||Jrb.Dk(a.k,b)==52){if(Jrb.Gn(a.k,b)==2)return 11;if(Jrb.Gn(a.k,b)==3){if(Jrb.mk(a.k,b)==1)return 12;c&&Jrb.Sl(a.k,b,1);return 5;}}else if(Jrb.Dk(a.k,b)==6){if(Jrb.mk(a.k,b)==-1)return Jrb.rn(a.k,b)==3?16:Jrb.sn(a.k,b)==3?15:14;c&&Jrb.Sl(a.k,b,-1);return Jrb.rn(a.k,b)!=Jrb.sn(a.k,b)?2:3;}return 0;};Jrb.Zd=function Zd(a,b,c){if(a.f[b])return 0;if(a.a){if(Jrb.rn(a.k,b)!=3)return 0;}else {if(Jrb.rn(a.k,b)>3)return 0;}if(Jrb.Dk(a.k,b)==6){c&&Jrb.Sl(a.k,b,1);return 2;}if(Jrb.Dk(a.k,b)==5){return 4;}return 0;};Jrb.$d=function $d(a,b,c){if(Jrb.mk(a.k,b)!=0)return 0;if(a.a){if(Jrb.Dk(a.k,b)==5){if(Jrb.ao(a.k,b)!=2)return 0;c&&Jrb.Sl(a.k,b,1);return 1;}if(Jrb.Dk(a.k,b)==7){if(Jrb.ao(a.k,b)!=2)return 0;c&&Jrb.Sl(a.k,b,-1);return Jrb.be(a,b)?6:3;}if(Jrb.Dk(a.k,b)==8){if(Jrb.ao(a.k,b)!=1)return 0;c&&Jrb.Sl(a.k,b,-1);return Jrb.be(a,b)?7:4;}if(Jrb.Dk(a.k,b)==16){if(Jrb.ao(a.k,b)!=1)return 0;c&&Jrb.Sl(a.k,b,-1);return Jrb.be(a,b)?5:2;}if(Jrb.Dk(a.k,b)==34){if(Jrb.ao(a.k,b)!=1)return 0;c&&Jrb.Sl(a.k,b,-1);return Jrb.be(a,b)?4:1;}}else {if(Jrb.Dk(a.k,b)==5){if(Jrb.ao(a.k,b)>2)return 0;c&&Jrb.Sl(a.k,b,1);return 1;}if(Jrb.Dk(a.k,b)==7){if(Jrb.ao(a.k,b)>2)return 0;c&&Jrb.Sl(a.k,b,-1);return Jrb.be(a,b)?5:3;}if(Jrb.Dk(a.k,b)==8){if(Jrb.ao(a.k,b)>1)return 0;c&&Jrb.Sl(a.k,b,-1);return Jrb.be(a,b)?7:4;}if(Jrb.Dk(a.k,b)==16){if(Jrb.ao(a.k,b)>1)return 0;c&&Jrb.Sl(a.k,b,-1);return Jrb.be(a,b)?5:2;}}return 0;};Jrb._d=function _d(a,b,c){var d,e,f;d=Jrb.Dk(a.k,b);if(d>=5&&d<=8||d==15||d==16||d==33||d==34||d==52){f=Jrb.Xn(a.k,b);if(f!=0)return true;e=Jrb.mk(a.k,b);if(d==5&&e>=0){c&&Jrb.Sl(a.k,b,e-1);return true;}if(d!=5&&e<=0){c&&Jrb.Sl(a.k,b,e+1);return true;}}return false;};Jrb.ae=function ae(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o;for(b=0;b<a.k.f;b++){Jrb.ou(a.k,1);if(a.d[b]){g=false;h=tX(Drb.NY,Qsb,6,a.k.f,15,1);o=tX(Drb.NY,Qsb,6,a.k.f,15,1);i=tX(Drb.NY,Qsb,6,a.k.f,15,1);h[0]=b;o[b]=-1;i[b]=1;e=0;j=0;while(e<=j&&!g){f=h[e];for(k=0;k<Jrb.Gn(a.k,f)&&!g;k++){l=(i[f]&1)==1^Jrb.Qk(a.k,Jrb.Hn(a.k,f,k))>1;d=Jrb.Fn(a.k,f,k);if(i[d]==0&&l){if(a.d[d]){if((i[f]&1)==1){a.d[b]=false;a.d[d]=false;a.n+=2;n=f;for(m=0;m<i[f];m++){c=Jrb.Dn(a.k,d,n);if(a.e[c]){a.e[c]=false;--a.c;}Jrb.pm(a.k,c,Jrb.Qk(a.k,c)==1?2:Jrb.Qk(a.k,c)-1);d=n;n=o[d];}g=true;}}else {h[++j]=d;o[d]=f;i[d]=i[f]+1;}}}++e;}}}};Jrb.be=function be(a,b){var c;for(c=0;c<Jrb.Gn(a.k,b);c++)if(Jrb.tl(a.k,Jrb.Fn(a.k,b,c)))return true;return false;};Jrb.ce=function ce(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r;if(b!=null){a.e=b;}else {a.e=tX(Drb.c3,itb,6,a.k.g,16,1);for(m=0;m<a.k.g;m++){if(Jrb.Tk(a.k,m)==8){a.e[m]=true;Jrb.tm(a.k,m,1);}}}for(n=0;n<a.k.g;n++)a.e[n]&&++a.c;a.d=tX(Drb.c3,itb,6,a.k.f,16,1);a.f=tX(Drb.c3,itb,6,a.k.f,16,1);for(h=0;h<a.k.f;h++){o=0;for(p=0;p<Jrb.Gn(a.k,h);p++)a.e[Jrb.Hn(a.k,h,p)]&&++o;if(o>0){a.d[h]=true;++a.b;o==3&&(a.f[h]=true);}}a.i=tX(Drb.c3,itb,6,c.i.a.length,16,1);for(r=0;r<c.i.a.length;r++){a.i[r]=true;for(j=Krb.Ji(c.j,r),k=0,l=j.length;k<l;++k){i=j[k];if(!a.e[i]){a.i[r]=false;break;}}}a.j=tX(Drb.c3,itb,6,a.k.f,16,1);a.g=tX(Drb.c3,itb,6,a.k.f,16,1);for(q=0;q<c.i.a.length;q++){if(a.i[q]&&Krb.Ji(c.j,q).length!=6){for(e=Krb.Ji(c.i,q),f=0,g=e.length;f<g;++f){d=e[f];Krb.Ji(c.j,q).length==5?a.g[d]=true:a.j[d]=true;}}}a.n=0;};Jrb.de=function de(a,b,c){var d,e,f,g;Jrb.ou(a.k,1);g=new Jrb.Tr(a.k,1);Jrb.ce(a,b,g);if(a.c==0)return true;if(c)for(e=0;e<a.k.f;e++)a.d[e]&&Jrb.Dk(a.k,e)==7&&Jrb.Gn(a.k,e)==3&&Jrb.Sl(a.k,e,1);a.a=false;Jrb.oe(a,c);a.k.K&&Jrb.ge(a);Jrb.pe(a,g);Jrb.le(a,g);Jrb.ne(a);Jrb.he(a);while(Jrb.je(a,g,c))Jrb.he(a);Jrb.ke(a,g,c)&&Jrb.he(a);while(Jrb.ie(a,g,c))Jrb.he(a);while(a.c!=0){for(f=0;f<a.k.g;f++){if(a.e[f]){Jrb.fe(a,f);Jrb.he(a);}}}a.b-a.n>=2&&Jrb.ae(a);for(d=0;d<a.k.f;d++){if(a.d[d]&&Jrb.Wn(a.k,d)!=0){c&&(Jrb.mk(a.k,d)==1&&Jrb.ql(a.k,d)||Jrb.mk(a.k,d)==-1&&Jrb.Dk(a.k,d)==5)?Jrb.Sl(a.k,d,0):Jrb.dm(a.k,d,32);++a.n;}}return a.b==a.n;};Jrb.ee=function ee(a,b){return a==b-1?0:a+1;};Jrb.fe=function fe(a,b){var c,d,e,f;if(Jrb.Tk(a.k,b)==1){Jrb.tm(a.k,b,2);a.n+=2;}for(e=0;e<2;e++){c=Jrb.Ik(a.k,e,b);a.d[c]=false;for(f=0;f<Jrb.Gn(a.k,c);f++){d=Jrb.Hn(a.k,c,f);if(a.e[d]){a.e[d]=false;--a.c;}}}};Jrb.ge=function ge(a){var b,c,d,e,f,g,h,i;for(c=0;c<a.k.g;c++){if(a.e[c]){for(e=0;e<2;e++){h=Jrb.Ik(a.k,e,c);b=false;for(g=0;g<Jrb.Gn(a.k,h);g++){if(c!=Jrb.Hn(a.k,h,g)&&a.e[Jrb.Hn(a.k,h,g)]){b=true;break;}}if(!b){i=c;d=Jrb.Ik(a.k,1-e,c);while(i!=-1){a.d[h]=false;a.e[i]=false;--a.c;Jrb.tm(a.k,i,8);i=-1;h=d;for(f=0;f<Jrb.Gn(a.k,h);f++){if(a.e[Jrb.Hn(a.k,h,f)]){if(i==-1){i=Jrb.Hn(a.k,h,f);d=Jrb.Fn(a.k,h,f);}else {h=-1;i=-1;break;}}}}h!=-1&&(a.d[d]=false);break;}}}}};Jrb.he=function he(a){var b,c,d,e,f,g,h;do{h=false;for(c=0;c<a.k.g;c++){if(a.e[c]){f=false;for(e=0;e<2;e++){d=Jrb.Ik(a.k,e,c);b=false;for(g=0;g<Jrb.Gn(a.k,d);g++){if(c!=Jrb.Hn(a.k,d,g)&&a.e[Jrb.Hn(a.k,d,g)]){b=true;break;}}if(!b){f=true;break;}}if(f){h=true;Jrb.fe(a,c);}}}}while(h);};Jrb.ie=function ie(a,b,c){var d,e,f,g,h,i,j,k,l,m;for(k=0;k<b.i.a.length;k++){if(Krb.Ji(b.j,k).length!=6&&a.i[k]){i=true;m=Krb.Ji(b.j,k);for(h=0;h<m.length;h++){if(!a.e[m[h]]){i=false;break;}}if(i){e=-1;f=0;l=Krb.Ji(b.i,k);for(g=0;g<l.length;g++){d=l[g];j=a.g[d]?Jrb.Yd(a,d,false):Jrb.Zd(a,d,false);if(f<j){f=j;e=g;}}if(e!=-1){d=l[e];c&&(a.g[d]?Jrb.Yd(a,d,true):Jrb.Zd(a,d,true));Jrb.me(a,d);return true;}}}}return false;};Jrb.je=function je(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;s=tX(Drb.NY,Qsb,6,a.k.g,15,1);for(p=0;p<b.i.a.length;p++){r=Krb.Ji(b.j,p);l=true;for(i=0;i<r.length;i++){if(!a.e[r[i]]){l=false;break;}}if(l)for(h=0;h<r.length;h++)++s[r[h]];}e=a.c;for(o=0;o<b.i.a.length;o++){g=false;q=Krb.Ji(b.i,o);r=Krb.Ji(b.j,o);for(h=0;h<r.length&&!g;h++){if(s[r[h]]>1){f=Jrb.ee(h,r.length);if(s[r[f]]==1){j=false;n=Jrb.ee(f,r.length);while(s[r[n]]==1){(!Jrb._d(a,q[n],false)||r.length!=6&&Jrb.Dk(a.k,q[n])!=6)&&(j=true);n=Jrb.ee(n,r.length);}if(!j){d=n>f?n-f:n+r.length-f;if(d>2&&(d&1)==1){for(m=1;m<d;m+=2){k=f+m<r.length?f+m:f+m-r.length;if(c){Jrb._d(a,q[k],true);Jrb._d(a,q[k==q.length-1?0:k+1],true);}Jrb.fe(a,r[k]);}g=true;}}}}}}return e!=a.c;};Jrb.ke=function ke(a,b,c){var d,e,f,g,h,i,j,k;d=false;for(i=0;i<b.i.a.length;i++){if(Krb.Ji(b.j,i).length==6){h=true;j=Krb.Ji(b.i,i);k=Krb.Ji(b.j,i);for(f=0;f<6;f++){if(!Jrb._d(a,j[f],false)||!a.e[k[f]]){h=false;break;}}if(h){if(c)for(g=0;g<6;g++)Jrb._d(a,j[g],true);for(e=0;e<6;e+=2)Jrb.fe(a,k[e]);d=true;break;}}}return d;};Jrb.le=function le(a,b){var c;for(c=0;c<a.k.g;c++){if(a.e[c]&&Jrb.Mr(b,c)){Jrb.me(a,Jrb.Ik(a.k,0,c));Jrb.me(a,Jrb.Ik(a.k,1,c));}}};Jrb.me=function me(a,b){var c,d;if(a.d[b]){a.d[b]=false;--a.b;}for(d=0;d<Jrb.Gn(a.k,b);d++){c=Jrb.Hn(a.k,b,d);if(a.e[c]){a.e[c]=false;--a.c;}}};Jrb.ne=function ne(a){var b,c,d,e,f;for(c=0;c<a.k.g;c++){if(Jrb.Qk(a.k,c)==2){for(e=0;e<2;e++){b=Jrb.Ik(a.k,e,c);if(Jrb.Dk(a.k,b)<=8){for(f=0;f<Jrb.Gn(a.k,b);f++){d=Jrb.Hn(a.k,b,f);if(a.e[d]){Jrb.me(a,b);break;}}}}}}};Jrb.oe=function oe(a,b){var c;for(c=0;c<a.k.f;c++)a.d[c]&&Jrb.Xn(a.k,c)==0&&(!b||Jrb.Dk(a.k,c)==5&&Jrb.mk(a.k,c)<0||Jrb.Dk(a.k,c)==6||Jrb.Dk(a.k,c)==14||Jrb.ql(a.k,c)&&Jrb.mk(a.k,c)>0)&&Jrb.me(a,c);};Jrb.pe=function pe(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G;t=tX(Drb.c3,itb,6,a.k.f,16,1);for(C=0;C<b.i.a.length;C++){D=Krb.Ji(b.j,C).length;if(D==3||D==5||D==6||D==7){if(a.i[C]){for(d=Krb.Ji(b.i,C),e=0,f=d.length;e<f;++e){c=d[e];t[c]=true;}}}}m=tX(Drb.NY,Qsb,6,a.k.f,15,1);o=tX(Drb.c3,itb,6,a.k.f,16,1);for(j=0;j<a.k.g;j++){g=Jrb.Ik(a.k,0,j);h=Jrb.Ik(a.k,1,j);if(!t[g]&&!t[h]){if(a.e[j]){++m[g];++m[h];}if(Jrb.Tk(a.k,j)==16){o[g]=true;o[h]=true;}}}B=tX(Drb.NY,Qsb,6,a.k.f,15,1);n=tX(Drb.NY,Qsb,6,a.k.f,15,1);for(G=0;G<a.k.f;G++){if(m[G]==1){n[0]=G;l=0;p=0;while(l<=p){for(q=0;q<Jrb.Gn(a.k,n[l]);q++){if(a.e[Jrb.Hn(a.k,n[l],q)]){k=Jrb.Fn(a.k,n[l],q);if((l==0||k!=n[l-1])&&m[k]!=0){n[++p]=k;if((m[k]&1)!=0){for(v=1;v<p;v+=2)B[n[v]]=-1;p=0;}break;}}}++l;}}}i=tX(Drb.c3,itb,6,a.k.f,16,1);for(F=0;F<a.k.f;F++){if(!i[F]&&m[F]!=0){n[0]=F;i[F]=true;l=0;p=0;while(l<=p){for(q=0;q<Jrb.Gn(a.k,n[l]);q++){if(a.e[Jrb.Hn(a.k,n[l],q)]){k=Jrb.Fn(a.k,n[l],q);if(!i[k]){n[++p]=k;i[k]=true;}}}++l;}if((p&1)==0){for(r=0;r<=p;r++)B[n[r]]==0&&(B[n[r]]=Jrb.$d(a,n[r],false));u=true;for(s=0;s<=p;s++){if(B[n[s]]<=0){if(!Jrb._d(a,n[s],false)){u=false;break;}}}if(u){A=0;w=-1;for(q=0;q<=p;q++){if(A<B[n[q]]){A=B[n[q]];w=n[q];}}if(A>0){Jrb.$d(a,w,true);Jrb.me(a,w);}}}}}};Jrb.qe=function qe(a){this.k=a;};Z3(86,1,{},Jrb.qe);_.a=false;_.b=0;_.c=0;_.n=0;Drb.UY=B5(86);Jrb.re=function re(a,b){var c,d;c=0;for(d=0;d<a.j[b];d++)a.n[b][d]==2&&(Jrb.Dk(a,a.i[b][d])==7||Jrb.Dk(a,a.i[b][d])==8||Jrb.Dk(a,a.i[b][d])==16)&&++c;return c;};Jrb.se=function se(a,b){var c,d,e,f,g,h;if(a.s[b]==0){return false;}h=true;c=a.s[b];f=a.j[b];g=0;for(d=0;d<f;d++){e=a.i[b][d];g+=a.s[e];}$wnd.Math.abs(c)<=$wnd.Math.abs(g)&&Erb.T6(c)!=Erb.T6(g)&&(h=false);return h;};Jrb.te=function te(a,b){var c,d,e,f,g,h,i;if(a.C[b]!=8||a.s[b]!=0||a.j[b]!=1||a.n[b][0]!=1)return false;c=a.i[b][0];if(a.C[c]==6){g=a.j[c];for(d=0;d<g;d++){e=a.i[c][d];if(e==b){continue;}if(a.C[e]!=8){continue;}f=Jrb.Dn(a,c,e);if(a.H[f]==2)return true;}}else if(a.C[c]==7){if(a.s[c]==1)return true;}else if(a.C[c]==16){h=a.j[c];i=0;for(d=0;d<h;d++){e=a.i[c][d];if(e==b)continue;if(a.C[e]!=8)continue;f=Jrb.Dn(a,c,e);a.H[f]==2&&++i;}if(i==2)return true;}else if(Jrb.ue(a,b))return true;return false;};Jrb.ue=function ue(a,b){var c,d,e,f,g;if(a.C[b]!=8)return false;if(a.j[b]!=1)return false;c=a.i[b][0];if(a.C[c]==15){g=a.j[c];for(d=0;d<g;d++){e=a.i[c][d];if(e==b)continue;if(a.C[e]!=8)continue;f=Jrb.Dn(a,c,e);if(a.H[f]==2)return true;}}return false;};Jrb.ve=function ve(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;if(a.C[b]!=7||a.s[b]!=0||a.j[b]+a.o[b]>3)return false;if(b<a.f&&Jrb.Fr(a.p,b)){if(a.o[b]!=1)return false;if(Jrb.zn(a,b,7)!=1)return false;u=(Jrb.ou(a,7),a.p);for(s=0;s<u.i.a.length;s++){if(Jrb.Hr(u,s,b)){if(Krb.Ji(u.j,s).length==5||Krb.Ji(u.j,s).length==6){v=Krb.Ji(u.i,s);q=-1;for(i=0;i<v.length;i++){if(v[i]==b){q=i;break;}}e=0;r=null;p=null;if(v.length==5){r=tX(Drb.NY,Qsb,6,2,15,1);r[0]=v[q-1<0?q+4:q-1];r[1]=v[q-4<0?q+1:q-4];p=tX(Drb.NY,Qsb,6,2,15,1);p[0]=v[q-2<0?q+3:q-2];p[1]=v[q-3<0?q+2:q-3];}if(v.length==6){r=tX(Drb.NY,Qsb,6,3,15,1);r[0]=v[q-1<0?q+5:q-1];r[1]=v[q-3<0?q+3:q-3];r[2]=v[q-5<0?q+1:q-5];p=tX(Drb.NY,Qsb,6,2,15,1);p[0]=v[q-2<0?q+4:q-2];p[1]=v[q-4<0?q+2:q-4];}for(j=0;j<v.length;j++)b!=v[j]&&Jrb.Dk(a,v[j])==7&&Jrb.wn(a,v[j])==1&&--e;for(k=0;k<r.length;k++){f=-1;g=-1;for(o=0;o<Jrb.Gn(a,r[k]);o++){if(!Jrb.zo(a,Jrb.Hn(a,r[k],o))){f=Jrb.Fn(a,r[k],o);g=Jrb.Hn(a,r[k],o);break;}}if(f!=-1){if(a.C[f]==7&&a.o[f]==0&&a.j[f]+a.o[f]<=3&&!Jrb.we(a,f,false)){++e;continue;}if(a.C[f]==8&&a.j[f]==1){e+=2;continue;}if(g<a.g&&Jrb.Gr(a.p,g)){for(w=0;w<u.i.a.length;w++){if(u.d[w]&&Jrb.Hr(u,w,f)){t=Krb.Ji(u.i,w);for(n=0;n<t.length;n++){if(Jrb.Dk(a,t[n])==7&&Jrb.wn(a,t[n])==1){--e;break;}}break;}}}}}for(l=0;l<p.length;l++){f=-1;for(n=0;n<Jrb.Gn(a,p[l]);n++)Jrb.zo(a,Jrb.Hn(a,p[l],n))||(f=Jrb.Fn(a,p[l],n));Jrb.Dk(a,p[l])==6?f!=-1&&Jrb.re(a,f)!=0&&--e:Jrb.Dk(a,p[l])==7&&Jrb.wn(a,p[l])==0&&(f==-1||!(f<a.f&&Jrb.Fr(a.p,f))&&Jrb.re(a,f)==0)&&++e;}return e>0;}break;}}return false;}if(a.o[b]>1)return false;if(a.o[b]==1){m=-1;A=0;for(i=0;i<a.j[b];i++){d=a.i[b][i];if(a.n[b][i]==2){if(a.C[d]!=6)return false;m=d;continue;}if(a.C[d]==8)return false;if(a.C[d]==7){--A;Jrb.we(a,d,false)&&--A;continue;}d<a.f&&Jrb.Fr(a.p,d)&&--A;}if(m==-1)return false;c=0;for(j=0;j<a.j[m];j++){if(a.n[m][j]==1){d=a.i[m][j];if(Jrb.re(a,d)!=0)return false;d<a.f&&Jrb.Fr(a.p,d)&&++c;a.C[d]==7&&!Jrb.we(a,d,true)&&++A;(a.C[d]==8||a.C[d]==16)&&--A;}}c==2&&--A;return A>=0;}for(h=0;h<a.j[b];h++){d=a.i[b][h];if(d<a.f&&Jrb.Fr(a.p,d))return false;if(a.C[d]!=6)return false;if(Jrb.re(a,d)!=0)return false;if(a.o[d]!=0&&Jrb.xe(a,d))return false;}return true;};Jrb.we=function we(a,b,c){var d,e,f;d=false;for(f=0;f<a.j[b];f++){if(!Jrb.zo(a,a.k[b][f])&&a.n[b][f]==1){e=a.i[b][f];if(!(e<a.f&&Jrb.Fr(a.p,e))&&(a.C[e]==6&&Jrb.re(a,e)==1||a.C[e]==16&&Jrb.re(a,e)==2)){if(d||!c)return true;d=true;}}}return false;};Jrb.xe=function xe(a,b){var c,d,e;for(d=0;d<a.j[b];d++){if(a.n[b][d]!=1){c=a.i[b][d];for(e=0;e<a.j[c];e++)if(a.n[c][e]==1&&Jrb.re(a,a.i[c][e])!=0)return true;}}return false;};Jrb.Ae=function Ae(){Jrb.Ae=_3;Jrb.ye=BX(nX(Drb.b3,1),hub,6,15,[-1,-1,-1,0,0,1,2,3,4,5,-1,0,0,0,6,7,8,9,-1,0,0,10,10,10,10,10,10,10,10,10,10,1,11,11,12,13,-1,0,0,10,10,10,10,10,10,10,10,10,10,0,0,0,11,14,-1,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,10,10,10,10,10,10,10,10,1,1,1,1,-1,-1,-1,-1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);Jrb.ze=BX(nX(Drb.b3,1),hub,6,15,[-1,-1,-1,0,0,0,2,5,5,5,-1,0,0,0,0,9,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);};Jrb.Be=function Be(a,b,c){Jrb.Ae();var d,e,f,g,h,i,j,k,l,m,n,o,p,q;Jrb.ou(a,7);n=tX(Drb.OY,iub,6,a.j[b],14,1);m=0;for(i=0;i<a.j[b];i++){e=a.i[b][i];if(a.C[e]==1)continue;g=0;if((c&32)!=0){f=v3(a.n[b][i]);c==32190?o3(f,3)<0&&Jrb.Do(a,a.k[b][i])&&a.o[b]==1&&(f=0):o3(f,3)<0&&Jrb.zo(a,a.k[b][i])&&(f=0);g=G3(g,H3(f,4));}if((c&128)!=0){if(Jrb.ye[a.C[e]]==-1)throw h3(new Erb._z(jub+a.C[e]));g=i3(g,v3(Jrb.ye[a.C[e]]));}else if((c&64)!=0){if(Jrb.ze[a.C[e]]==-1)throw h3(new Erb._z(jub+a.C[e]));g=i3(g,v3(Jrb.ze[a.C[e]]));}if((c&256)!=0){o=a.j[e]-1;o>3&&(o=3);(c&512)==0&&o>1&&(o=1);g=G3(g,Drb.r3(o<<6));}(c&vtb)!=0&&(a.u[e]&8)!=0&&(g=G3(g,256));(c&Vtb)!=0&&e<a.f&&Jrb.Fr(a.p,e)&&(g=G3(g,512));(c&ptb)!=0&&(a.u[e]&Utb)!=0&&(g=G3(g,vtb));k=0;while(o3(g,n[k])<0)++k;for(l=i;l>k;l--)n[l]=n[l-1];n[k]=g;++m;}m>4&&(m=4);d=0;for(j=0;j<m;j++){d=H3(d,11);d=G3(d,n[j]);}d=H3(d,15);if(Jrb.ye[a.C[b]]==-1)throw h3(new Erb._z(jub+a.C[b]));d=G3(d,v3(Jrb.ye[a.C[b]]));if((c&2)!=0){q=!!a.p&&b<a.f?Jrb.ur(a.p,b):0;q>9&&(q=9);q>2&&(q-=2);d=G3(d,Drb.r3(q<<4));}else (c&1)!=0&&(a.u[b]&8)!=0&&(d=G3(d,64));(c&4)!=0&&b<a.f&&Jrb.Fr(a.p,b)&&(d=G3(d,vtb));(c&8)!=0&&(a.u[b]&qtb)!=0&&(d=G3(d,Vtb));(c&16)!=0&&(a.u[b]&Utb)!=0&&(d=G3(d,qtb));if((c&qtb)!=0){Jrb.se(a,b)&&(d=G3(d,Utb));if(Jrb.ve(a,b)){for(h=0;h<a.f;h++){if(Jrb.te(a,h)){d=G3(d,ptb);break;}}}}if((c&Utb)!=0){p=v3(Jrb.zn(a,b,10));d=G3(d,H3(p,7));}return d;};Jrb.Ce=function Ce(a,b,c,d,e,f){var g,h,i,j;i=0;for(h=0;h<a.Q.f;h++)E3(j3(Jrb.yk(a.Q,a.A[h]),d),0)&&++i;if(i==0)return;Jrb.Af(a,b);Jrb.vf(a,Drb.r3(i),c);for(g=0;g<a.Q.f;g++){j=j3(Jrb.yk(a.Q,a.A[g]),d);if(o3(j,0)!=0){Jrb.vf(a,Drb.r3(g),c);e!=1&&Jrb.vf(a,I3(j,f),e);}}};Jrb.De=function De(a,b,c,d,e,f){var g,h,i,j;i=0;for(h=0;h<a.Q.g;h++)(Jrb.Sk(a.Q,a.B[h])&d)!=0&&++i;if(i==0)return;Jrb.Af(a,b);Jrb.vf(a,Drb.r3(i),c);for(g=0;g<a.Q.g;g++){j=Jrb.Sk(a.Q,a.B[g])&d;if(j!=0){Jrb.vf(a,Drb.r3(g),c);e!=1&&Jrb.vf(a,Drb.r3(j>>f),e);}}};Jrb.Ee=function Ee(a){var b,c,d,e,f,g;while(true){f=tX(Drb.c3,itb,6,a.T+1,16,1);d=-1;for(c=0;c<a.Q.f;c++){if(Jrb.Xn(a.Q,c)!=0){f[a.e[c]]&&d<a.e[c]&&(d=a.e[c]);f[a.e[c]]=true;}}if(d==-1)break;e=0;for(b=0;b<a.Q.f;b++){g=0;a.e[b]==d&&(g=++e);Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));Jrb.tg(a.d[b],8,Drb.r3(g));}a.T=Jrb.df(a);}};Jrb.Fe=function Fe(a){var b,c,d,e;for(c=0;c<a.Q.f;c++){Jrb.wg(a.d[c],c);Jrb.tg(a.d[c],2*a.c+4,H3(v3(a.e[c]),a.c+4));}e=false;for(b=0;b<a.Q.f;b++)e=e|Jrb.Se(a,b,3);for(d=0;d<a.Q.g;d++)e=e|Jrb.Ne(a,d,3);if(!e)return false;while(a.T<a.Q.f){e=Jrb.af(a);if(!e)break;Jrb.cf(a);!!a.O&&Jrb.Mg(a.O,a.e);}return true;};Jrb.Ge=function Ge(a){var b,c,d,e,f;for(c=0;c<a.Q.f;c++){Jrb.wg(a.d[c],c);Jrb.tg(a.d[c],a.c+1,C3(2,v3(a.e[c])));}f=tX(Drb.NY,Qsb,6,a.T+1,15,1);for(d=0;d<a.Q.f;d++)++f[a.e[d]];e=1;while(f[e]==1)++e;for(b=0;b<a.Q.f;b++){if(a.e[b]==e){Jrb.ug(a.d[b],1);break;}}a.T=Jrb.df(a);};Jrb.He=function He(a,b,c){var d,e,f,g,h;if(Jrb.Dk(a.Q,b)!=6&&Jrb.Dk(a.Q,b)!=7)return false;e=Jrb.Fn(a.Q,b,0);f=Jrb.Fn(a.Q,b,1);if(Jrb.wn(a.Q,e)!=1||Jrb.wn(a.Q,f)!=1)return false;if(Jrb.Gn(a.Q,e)==1||Jrb.Gn(a.Q,f)==1)return false;if(Jrb.rn(a.Q,e)>3||Jrb.rn(a.Q,f)>3)return false;g=new Jrb.wj(a.Q,a.e,b,e);if(g.f&&c==1)return false;h=new Jrb.wj(a.Q,a.e,b,f);if(h.f&&c==1)return false;if(g.f&&h.f)return false;if(c==3){g.f&&g.c&&(a.V[b]=true);h.f&&h.c&&(a.V[b]=true);}d=a.ib?Jrb.Je(a,g,h):Jrb.Ie(g,h);if(c==1){a.cb[b]=d;}else if(c==2){g.f&&(d==1?Jrb.ug(a.d[g.b],v3(a.e[e])):Jrb.ug(a.d[g.d],v3(a.e[e])));h.f&&(d==2?Jrb.ug(a.d[h.b],v3(a.e[f])):Jrb.ug(a.d[h.d],v3(a.e[f])));}return true;};Jrb.Ie=function Ie(a,b){var c,d,e;d=Jrb.vj(a);e=Jrb.vj(b);if(d==-1||e==-1||(d+e&1)==0)return 3;c=0;switch(d+e){case 3:case 7:c=2;break;case 5:c=1;}return c;};Jrb.Je=function Je(a,b,c){var d,e;d=tX(Drb.NY,Qsb,6,4,15,1);d[0]=b.b;d[1]=b.a;d[2]=c.a;d[3]=c.b;e=Jrb.Sj(a.Q,d);if($wnd.Math.abs(e)<0.3||$wnd.Math.abs(e)>kub)return 3;return e<0?2:1;};Jrb.Ke=function Ke(a,b,c){var d,e,f,g,h;if(!Jrb.Ao(a.Q,b))return false;d=Jrb.Ik(a.Q,0,b);e=Jrb.Ik(a.Q,1,b);g=new Jrb.wj(a.Q,a.e,d,e);if(g.f&&c==1)return false;h=new Jrb.wj(a.Q,a.e,e,d);if(h.f&&c==1)return false;if(g.f&&h.f)return false;if(c==3){g.f&&(a.U[b]=Jrb.Qf(a,e));h.f&&(a.U[b]=Jrb.Qf(a,d));}f=a.ib?Jrb.Me(a,g,h):Jrb.Le(g,h);if(c==1){a.n[b]=f;}else if(c==2){g.f&&(f==2?Jrb.ug(a.d[g.b],v3(a.e[e])):Jrb.ug(a.d[g.d],v3(a.e[e])));h.f&&(f==2?Jrb.ug(a.d[h.b],v3(a.e[d])):Jrb.ug(a.d[h.d],v3(a.e[d])));}return true;};Jrb.Le=function Le(a,b){var c,d,e;d=Jrb.vj(a);e=Jrb.vj(b);if(d==-1||e==-1||(d+e&1)==0)return 3;c=0;switch(d+e){case 3:case 7:c=1;break;case 5:c=2;}return c;};Jrb.Me=function Me(a,b,c){var d,e;d=tX(Drb.NY,Qsb,6,4,15,1);d[0]=b.b;d[1]=b.a;d[2]=c.a;d[3]=c.b;e=Jrb.Sj(a.Q,d);if($wnd.Math.abs(e)<0.3||$wnd.Math.abs(e)>kub)return 3;return e<0?1:2;};Jrb.Ne=function Ne(a,b,c){var d,e,f,g,h,i,j,k;if(a.n[b]!=0)return false;if(a.Q.K){for(j=0;j<2;j++){d=Jrb.Ik(a.Q,j,b);for(k=0;k<Jrb.rn(a.Q,d);k++)if(E3(j3(Jrb.yk(a.Q,Jrb.Fn(a.Q,d,k)),jtb),0))return false;}}if(Jrb.Qk(a.Q,b)==1)return Jrb.Ke(a,b,c);if(Jrb.Qk(a.Q,b)!=2)return false;if(Jrb.zo(a.Q,b))return false;f=Jrb.Ik(a.Q,0,b);g=Jrb.Ik(a.Q,1,b);if(Jrb.Gn(a.Q,f)==1||Jrb.Gn(a.Q,g)==1)return false;if(Jrb.Gn(a.Q,f)>3||Jrb.Gn(a.Q,g)>3)return false;if(Jrb.wn(a.Q,f)==2||Jrb.wn(a.Q,g)==2)return false;h=new Jrb.wj(a.Q,a.e,g,f);if(h.f&&c==1)return false;i=new Jrb.wj(a.Q,a.e,f,g);if(i.f&&c==1)return false;if(h.f&&i.f)return false;if(c==3){h.f&&h.c&&(a.U[b]=true);i.f&&i.c&&(a.U[b]=true);}e=Jrb.pl(a.Q,b)?3:a.ib?Jrb.Pe(a,h,i):Jrb.Oe(h,i);if(c==1){a.n[b]=e;}else if(c==2){h.f&&(e==1?Jrb.ug(a.d[h.b],v3(a.e[f])):e==2&&Jrb.ug(a.d[h.d],v3(a.e[f])));i.f&&(e==1?Jrb.ug(a.d[i.b],v3(a.e[g])):e==2&&Jrb.ug(a.d[i.d],v3(a.e[g])));}return true;};Jrb.Oe=function Oe(a,b){if(Jrb.vj(a)==-1||Jrb.vj(b)==-1)return 3;if(((Jrb.vj(a)|Jrb.vj(b))&1)!=0)return 3;return Jrb.vj(a)==Jrb.vj(b)?1:2;};Jrb.Pe=function Pe(a,b,c){var d,e,f,g,h,i,j;f=tX(Drb.LY,Osb,6,3,15,1);f[0]=Jrb.Ak(a.Q,c.a)-Jrb.Ak(a.Q,b.a);f[1]=Jrb.Bk(a.Q,c.a)-Jrb.Bk(a.Q,b.a);f[2]=Jrb.Ck(a.Q,c.a)-Jrb.Ck(a.Q,b.a);i=tX(Drb.LY,Osb,6,3,15,1);i[0]=Jrb.Ak(a.Q,b.b)-Jrb.Ak(a.Q,b.a);i[1]=Jrb.Bk(a.Q,b.b)-Jrb.Bk(a.Q,b.a);i[2]=Jrb.Ck(a.Q,b.b)-Jrb.Ck(a.Q,b.a);j=tX(Drb.LY,Osb,6,3,15,1);j[0]=Jrb.Ak(a.Q,c.b)-Jrb.Ak(a.Q,c.a);j[1]=Jrb.Bk(a.Q,c.b)-Jrb.Bk(a.Q,c.a);j[2]=Jrb.Ck(a.Q,c.b)-Jrb.Ck(a.Q,c.a);g=tX(Drb.LY,Osb,6,3,15,1);g[0]=f[1]*i[2]-f[2]*i[1];g[1]=f[2]*i[0]-f[0]*i[2];g[2]=f[0]*i[1]-f[1]*i[0];h=tX(Drb.LY,Osb,6,3,15,1);h[0]=f[1]*g[2]-f[2]*g[1];h[1]=f[2]*g[0]-f[0]*g[2];h[2]=f[0]*g[1]-f[1]*g[0];d=(i[0]*h[0]+i[1]*h[1]+i[2]*h[2])/($wnd.Math.sqrt(i[0]*i[0]+i[1]*i[1]+i[2]*i[2])*$wnd.Math.sqrt(h[0]*h[0]+h[1]*h[1]+h[2]*h[2]));e=(j[0]*h[0]+j[1]*h[1]+j[2]*h[2])/($wnd.Math.sqrt(j[0]*j[0]+j[1]*j[1]+j[2]*j[2])*$wnd.Math.sqrt(h[0]*h[0]+h[1]*h[1]+h[2]*h[2]));return d<0^e<0?1:2;};Jrb.Qe=function Qe(a,b){var c,d,e,f;c=Jrb.kk(a.Q,b);d=Jrb.Un(a.Q,b,false);e=Jrb.Un(a.Q,b,true);f=-1;if(d!=e){c!=-1&&c>d?f=c<<24>>24:f=d<<24>>24;}else if(c!=-1){(c>e||c<e&&c>=Jrb.ao(a.Q,b))&&(f=c<<24>>24);}else if(!Jrb.ep(a.Q,b)&&Jrb.Kn(a.Q,b)!=0){f=Jrb.ao(a.Q,b);f-=Jrb.Xk(a.Q,b,f);}Jrb.lf(a,b,f);return f;};Jrb.Re=function Re(a){var b,c,d,e,f,g,h,i,j,k,l;d=tX(Drb.NY,Qsb,6,a.N,15,1);for(b=0;b<a.Q.f;b++){k=Jrb.Gn(a.Q,b)+Jrb.Zn(a.Q,b);j=0;for(f=0;f<Jrb.sn(a.Q,b);f++){if(f<Jrb.Gn(a.Q,b)||f>=Jrb.rn(a.Q,b)){l=2*a.e[Jrb.Fn(a.Q,b,f)];c=Jrb.Hn(a.Q,b,f);Jrb.Qk(a.Q,c)==2&&(Jrb.zo(a.Q,c)||++l);for(h=0;h<j;h++)if(l<d[h])break;for(i=j;i>h;i--)d[i]=d[i-1];d[h]=l;++j;}}Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));for(g=k;g<a.N;g++)Jrb.tg(a.d[b],a.c+1,0);for(e=0;e<k;e++)Jrb.tg(a.d[b],a.c+1,Drb.r3(d[e]));}};Jrb.Se=function Se(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;if(a.cb[b]!=0)return false;if(Jrb.Dk(a.Q,b)!=5&&Jrb.Dk(a.Q,b)!=6&&Jrb.Dk(a.Q,b)!=7&&Jrb.Dk(a.Q,b)!=14&&Jrb.Dk(a.Q,b)!=15&&Jrb.Dk(a.Q,b)!=16)return false;if(Jrb.wn(a.Q,b)!=0){if(Jrb.Bo(a.Q,b))return Jrb.He(a,b,c);if(Jrb.Dk(a.Q,b)!=15&&Jrb.Dk(a.Q,b)!=16)return false;}if(Jrb.Gn(a.Q,b)<3||Jrb.rn(a.Q,b)>4)return false;if(Jrb.mk(a.Q,b)>0&&Jrb.Dk(a.Q,b)==6)return false;if(Jrb.Dk(a.Q,b)==5&&Jrb.rn(a.Q,b)!=4)return false;if(a.Q.K){if(E3(j3(Jrb.yk(a.Q,b),jtb),0))return false;for(h=0;h<Jrb.rn(a.Q,b);h++)if(E3(j3(Jrb.yk(a.Q,Jrb.Fn(a.Q,b,h)),jtb),0))return false;}if(Jrb.Dk(a.Q,b)==7&&!a.R[b])return false;o=tX(Drb.NY,Qsb,6,4,15,1);p=tX(Drb.NY,Qsb,6,4,15,1);k=tX(Drb.c3,itb,6,4,16,1);for(i=0;i<Jrb.rn(a.Q,b);i++){f=-1;e=0;for(j=0;j<Jrb.rn(a.Q,b);j++){if(!k[j]){if(f<a.e[Jrb.Fn(a.Q,b,j)]){f=a.e[Jrb.Fn(a.Q,b,j)];e=j;}}}o[i]=e;p[i]=f;k[e]=true;}if(Jrb.rn(a.Q,b)==4&&p[0]==p[1]&&p[2]==p[3])return false;if(Jrb.rn(a.Q,b)==4&&(p[0]==p[2]||p[1]==p[3]))return false;if(Jrb.rn(a.Q,b)==3&&p[0]==p[2])return false;l=0;m=0;n=false;for(g=1;g<Jrb.rn(a.Q,b);g++){if(p[g-1]==p[g]){if(c==1||p[g]==0)return false;l=Jrb.Fn(a.Q,b,o[g-1]);m=Jrb.Fn(a.Q,b,o[g]);c==3&&Jrb.Mo(a.Q,Jrb.Hn(a.Q,b,o[g]))&&(a.V[b]=true);n=true;}}if(c!=1&&!n)return false;d=a.ib?Jrb.Ue(a,b,o):Jrb.Te(a,b,o);c==1?a.cb[b]=d:c==2&&(d==1?Jrb.ug(a.d[l],v3(a.e[b])):d==2&&Jrb.ug(a.d[m],v3(a.e[b])));return true;};Jrb.Te=function Te(a,b,c){var d,e,f,g,h,i,j,k,l,m;m=BX(nX(Drb.NY,2),lub,7,0,[BX(nX(Drb.NY,1),Qsb,6,15,[2,1,2,1]),BX(nX(Drb.NY,1),Qsb,6,15,[1,2,2,1]),BX(nX(Drb.NY,1),Qsb,6,15,[1,1,2,2]),BX(nX(Drb.NY,1),Qsb,6,15,[2,1,1,2]),BX(nX(Drb.NY,1),Qsb,6,15,[2,2,1,1]),BX(nX(Drb.NY,1),Qsb,6,15,[1,2,1,2])]);d=tX(Drb.LY,Osb,6,Jrb.rn(a.Q,b),15,1);for(g=0;g<Jrb.rn(a.Q,b);g++)d[g]=Jrb.Hk(a.Q,Jrb.Fn(a.Q,b,c[g]),b);j=Jrb.Mn(a.Q,b,c,d,null)<<24>>24;if(j!=3)return j;k=0;l=0;for(h=0;h<Jrb.rn(a.Q,b);h++){e=Jrb.Hn(a.Q,b,c[h]);if(Jrb.Ik(a.Q,0,e)==b){if(Jrb.Tk(a.Q,e)==129){l!=0&&Jrb.Cm(a.Q,b);k=h;l=1;}if(Jrb.Tk(a.Q,e)==257){l!=0&&Jrb.Cm(a.Q,b);k=h;l=2;}}}if(l==0)return 3;for(f=1;f<Jrb.rn(a.Q,b);f++)d[f]<d[0]&&(d[f]+=mtb);if(Jrb.rn(a.Q,b)==3){switch(k){case 0:(d[1]<d[2]&&d[2]-d[1]<ntb||d[1]>d[2]&&d[1]-d[2]>ntb)&&(l=3-l);break;case 1:d[2]-d[0]>ntb&&(l=3-l);break;case 2:d[1]-d[0]<ntb&&(l=3-l);}return l==1?2:1;}i=0;d[1]<=d[2]&&d[2]<=d[3]?i=0:d[1]<=d[3]&&d[3]<=d[2]?i=1:d[2]<=d[1]&&d[1]<=d[3]?i=2:d[2]<=d[3]&&d[3]<=d[1]?i=3:d[3]<=d[1]&&d[1]<=d[2]?i=4:d[3]<=d[2]&&d[2]<=d[1]&&(i=5);return m[i][k]==l?2:1;};Jrb.Ue=function Ue(a,b,c){var d,e,f,g,h,i;d=tX(Drb.NY,Qsb,6,4,15,1);for(h=0;h<Jrb.rn(a.Q,b);h++)d[h]=Jrb.Fn(a.Q,b,c[h]);Jrb.rn(a.Q,b)==3&&(d[3]=b);e=rX(Drb.LY,[ssb,Osb],[12,6],15,[3,3],2);for(g=0;g<3;g++){e[g][0]=Jrb.Ak(a.Q,d[g+1])-Jrb.Ak(a.Q,d[0]);e[g][1]=Jrb.Bk(a.Q,d[g+1])-Jrb.Bk(a.Q,d[0]);e[g][2]=Jrb.Ck(a.Q,d[g+1])-Jrb.Ck(a.Q,d[0]);}i=tX(Drb.LY,Osb,6,3,15,1);i[0]=e[0][1]*e[1][2]-e[0][2]*e[1][1];i[1]=e[0][2]*e[1][0]-e[0][0]*e[1][2];i[2]=e[0][0]*e[1][1]-e[0][1]*e[1][0];f=(e[2][0]*i[0]+e[2][1]*i[1]+e[2][2]*i[2])/($wnd.Math.sqrt(e[2][0]*e[2][0]+e[2][1]*e[2][1]+e[2][2]*e[2][2])*$wnd.Math.sqrt(i[0]*i[0]+i[1]*i[1]+i[2]*i[2]));return f>0?1:2;};Jrb.Ve=function Ve(a){var b,c;b=0;Krb.Web(a.d);for(c=0;c<a.d.length;c++){(c==0||Jrb.vg(a.d[c],a.d[c-1])!=0)&&++b;a.e[a.d[c].a]=b;}return b;};Jrb.We=function We(a){var b,c,d,e,f,g,h,i,j,k,l,m;if(a.w)return;a.w=new Krb.Yi();k=0;l=tX(Drb.NY,Qsb,6,a.Q.f,15,1);g=tX(Drb.NY,Qsb,6,a.Q.f,15,1);i=tX(Drb.NY,Qsb,6,a.Q.g,15,1);for(b=0;b<a.Q.f;b++){if(l[b]==0&&(Jrb.Lo(a.Q,b)||Jrb.wn(a.Q,b)==1)){g[0]=b;h=1;j=0;l[b]=++k;c=tX(Drb.c3,itb,6,a.Q.g,16,1);for(f=0;f<h;f++){for(m=0;m<Jrb.Gn(a.Q,g[f]);m++){e=Jrb.Hn(a.Q,g[f],m);if(Jrb.Mo(a.Q,e)||Jrb.Qk(a.Q,e)==2||Jrb.Ao(a.Q,e)){d=Jrb.Fn(a.Q,g[f],m);if(!c[e]){i[j++]=e;c[e]=true;}if(l[d]==0){g[h++]=d;l[d]=k;}}}}Krb.Ei(a.w,new Jrb.Cg(g,h,i,j));}}};Jrb.Xe=function Xe(a){var b;a.R=tX(Drb.c3,itb,6,a.Q.f,16,1);for(b=0;b<a.Q.f;b++){if(Jrb.Dk(a.Q,b)==7){if(Jrb.Gn(a.Q,b)==4){a.R[b]=true;continue;}if(Jrb.Gn(a.Q,b)==3){if(Jrb.An(a.Q,b)==3){a.R[b]=true;continue;}if(Jrb.mk(a.Q,b)==1){a.R[b]=true;continue;}if(Jrb.Fo(a.Q,b,true))continue;if((a.P&32)!=0){a.R[b]=true;continue;}if(Jrb.Ko(a.Q,b)){a.R[b]=true;continue;}}}}};Jrb.Ye=function Ye(a,b){var c,d,e,f;e=false;for(d=0;d<a.Q.g;d++)if(Jrb.Ne(a,d,1)){a.p[d]=a.K;b&&Jrb.mf(a,d);e=true;}f=false;for(c=0;c<a.Q.f;c++)if(Jrb.Se(a,c,1)){a.hb[c]=a.K;b&&Jrb.nf(a,c);f=true;}f&&(a.K=!a.K);return e||f;};Jrb.Ze=function Ze(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B;s=tX(Drb.c3,itb,6,a.Q.f,16,1);t=tX(Drb.c3,itb,6,a.Q.g,16,1);b=0;v=false;if((a.P&128)!=0){a.X=tX(Drb.NY,Qsb,6,a.Q.f,15,1);a.W=tX(Drb.NY,Qsb,6,a.Q.g,15,1);}for(d=0;d<a.Q.f;d++){if(a.V[d]){if(!a.eb[d]){if(Jrb.Se(a,d,1)){a.eb[d]=true;s[d]=true;++b;}}}}for(f=0;f<a.Q.g;f++){if(a.U[f]){if(!a.o[f]){if(Jrb.Ne(a,f,1)){a.o[f]=true;t[f]=true;++b;}}}}if(b==1){for(c=0;c<a.Q.f;c++){if(s[c]){a.cb[c]=0;break;}}for(e=0;e<a.Q.g;e++){if(t[e]){a.n[e]=0;break;}}}else if(b>1){Jrb.We(a);a.S=0;for(h=new Krb.ueb(a.w);h.a<h.c.a.length;){g=Krb.teb(h);u=0;w=0;k=0;j=0;l=-1;i=-1;for(o=0;o<g.a.length;o++){if(s[g.a[o]]){++u;if(a.cb[g.a[o]]==1||a.cb[g.a[o]]==2){++w;v=true;if(l<a.e[g.a[o]]){l=a.e[g.a[o]];k=g.a[o];}}}}for(p=0;p<g.b.length;p++){if(t[g.b[p]]){++u;A=a.e[Jrb.Ik(a.Q,0,g.b[p])];B=a.e[Jrb.Ik(a.Q,1,g.b[p])];m=A>B?(A<<16)+B:(B<<16)+A;if(a.n[g.b[p]]==1||a.n[g.b[p]]==2){++w;v=true;if(i<m){i=m;j=g.b[p];}}}}if(u==0)continue;if(u==1){for(q=0;q<g.a.length;q++)s[g.a[q]]&&(a.cb[g.a[q]]=0);for(n=0;n<g.b.length;n++)t[g.b[n]]&&(a.n[g.b[n]]=0);}else {if(w==1){for(q=0;q<g.a.length;q++)s[g.a[q]]&&(a.cb[g.a[q]]=3);for(n=0;n<g.b.length;n++)t[g.b[n]]&&(a.n[g.b[n]]=3);}else {if((a.P&128)!=0){++a.S;for(q=0;q<g.a.length;q++)s[g.a[q]]&&(a.X[g.a[q]]=a.S);for(n=0;n<g.b.length;n++)t[g.b[n]]&&(a.W[g.b[n]]=a.S);}r=false;l!=-1?a.cb[k]==2&&(r=true):a.n[j]==2&&(r=true);if(r){for(q=0;q<g.a.length;q++){if(s[g.a[q]]){switch(a.cb[g.a[q]]){case 1:a.cb[g.a[q]]=2;break;case 2:a.cb[g.a[q]]=1;}}}for(n=0;n<g.b.length;n++){if(t[g.b[n]]){switch(a.n[g.b[n]]){case 1:a.n[g.b[n]]=2;break;case 2:a.n[g.b[n]]=1;}}}}}}}}return v;};Jrb.$e=function $e(a,b){var c,d,e,f,g,h,i,j,k,l,m;f=rX(Drb.NY,[lub,Qsb],[7,6],15,[2,32],2);for(g=0;g<2;g++){c=tX(Drb.NY,lub,7,32,0,2);m=0;for(e=0;e<32;e++){if(b[g][e]!=null){k=b[g][e].length;c[e]=tX(Drb.NY,Qsb,6,k,15,1);for(h=0;h<k;h++)c[e][h]=a.e[b[g][e][h]];Hrb.Nmb(c[e],Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));++m;}}for(l=m;l>0;l--){j=0;i=null;for(d=0;d<32;d++){if(c[d]!=null){if(i==null||i.length<c[d].length){i=c[d];j=d;}else if(i.length==c[d].length){for(h=i.length-1;h>=0;h--){if(i[h]<c[d][h]){i=c[d];j=d;break;}}}}}f[g][j]=l;c[j]=null;}}return f;};Jrb._e=function _e(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t;l=false;if(a.Q.K){for(j=0;j<a.Q.g;j++){if(Jrb.Sk(a.Q,j)!=0){l=true;break;}}}a.N=2;for(c=0;c<a.Q.f;c++)a.N=$wnd.Math.max(a.N,Jrb.Gn(a.Q,c)+Jrb.Zn(a.Q,c));i=$wnd.Math.max(2,l?(62+a.c+a.N*(a.c+23))/63|0:(62+a.c+a.N*(a.c+5))/63|0);a.e=tX(Drb.NY,Qsb,6,a.Q.q,15,1);a.d=tX(Drb.ZY,mub,118,a.Q.f,0,1);for(d=0;d<a.Q.f;d++)a.d[d]=new Jrb.xg(i);h=false;for(e=0;e<a.Q.f;e++){Jrb.wg(a.d[e],e);E3(j3(Jrb.yk(a.Q,e),1),0)||Jrb.tk(a.Q,e)!=null?Jrb.tg(a.d[e],8,6):Jrb.tg(a.d[e],8,v3(Jrb.Dk(a.Q,e)));Jrb.tg(a.d[e],8,v3(Jrb.wk(a.Q,e)));Jrb.tg(a.d[e],2,v3(Jrb.wn(a.Q,e)));Jrb.tg(a.d[e],4,v3(Jrb.Gn(a.Q,e)+Jrb.Zn(a.Q,e)));E3(j3(Jrb.yk(a.Q,e),1),0)?Jrb.tg(a.d[e],4,8):Jrb.tg(a.d[e],4,v3(8+Jrb.mk(a.Q,e)));Jrb.tg(a.d[e],5,v3($wnd.Math.min(31,Jrb.An(a.Q,e))));Jrb.tg(a.d[e],4,v3(Jrb.Qe(a,e)+1));Jrb.tg(a.d[e],2,v3(Jrb.zk(a.Q,e)>>4));if(a.Q.K){Jrb.tg(a.d[e],46,Jrb.yk(a.Q,e));Jrb.tk(a.Q,e)!=null&&(h=true);}}a.T=Jrb.df(a);if(a.T<a.Q.f){for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));m=tX(Drb.NY,Qsb,6,Jrb.Gn(a.Q,b),15,1);for(o=0;o<Jrb.Gn(a.Q,b);o++){m[o]=a.e[Jrb.Fn(a.Q,b,o)]<<5;m[o]|=$wnd.Math.min(31,Jrb.En(a.Q,Jrb.Hn(a.Q,b,o)));}Hrb.Nmb(m,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));for(p=a.N;p>m.length;p--)Jrb.tg(a.d[b],a.c+5,0);for(n=m.length-1;n>=0;n--)Jrb.tg(a.d[b],a.c+5,Drb.r3(m[n]));}a.T=Jrb.df(a);}if(h&&a.T<a.Q.f){for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));g=Jrb.tk(a.Q,b);s=g==null?0:$wnd.Math.min(12,g.length);for(o=12;o>s;o--)Jrb.tg(a.d[b],8,0);for(n=s-1;n>=0;n--)Jrb.tg(a.d[b],8,Drb.r3(g[n]));}a.T=Jrb.df(a);}if(l&&a.T<a.Q.f){for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));k=tX(Drb.OY,iub,6,Jrb.Gn(a.Q,b)+Jrb.Zn(a.Q,b),14,1);q=0;for(o=0;o<Jrb.sn(a.Q,b);o++){if(o<Jrb.Gn(a.Q,b)||o>=Jrb.rn(a.Q,b)){k[q]=v3(a.e[Jrb.Fn(a.Q,b,o)]);k[q]=H3(k[q],23);k[q]=G3(k[q],v3(Jrb.Sk(a.Q,Jrb.Hn(a.Q,b,o))));++q;}}Hrb.Nmb(k,Hrb.rnb(a4(Krb.bfb.prototype.qc,Krb.bfb,[])));for(p=a.N;p>k.length;p--)Jrb.tg(a.d[b],a.c+23,0);for(n=k.length-1;n>=0;n--)Jrb.tg(a.d[b],a.c+23,k[n]);}a.T=Jrb.df(a);}if((a.P&8)!=0&&a.T<a.Q.f){r=new Jrb.ku();for(f=0;f<a.Q.f;f++)Jrb.ok(a.Q,f)!=null&&Jrb.iu(r,Jrb.ok(a.Q,f));for(b=0;b<a.Q.f;b++){t=Jrb.ok(a.Q,b)==null?0:1+Jrb.ju(r,Jrb.ok(a.Q,b));Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));Jrb.tg(a.d[b],a.c,Drb.r3(t));}a.T=Jrb.df(a);}if((a.P&16)!=0&&a.T<a.Q.f){for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));Jrb.tg(a.d[b],1,v3(Jrb.xl(a.Q,b)?1:0));}a.T=Jrb.df(a);}(a.P&512)!=0&&a.Q.K&&Jrb.Ee(a);};Jrb.af=function af(a){var b,c,d,e,f,g,h,i,j;for(c=0;c<a.Q.f;c++){Jrb.wg(a.d[c],c);Jrb.tg(a.d[c],2*a.c+4,H3(v3(a.e[c]),a.c+4));}for(i=1;i<=a.T;i++){e=false;for(d=0;d<a.Q.f;d++)a.e[d]==i&&(e=e|Jrb.Se(a,d,2));if(e){h=a.T;a.T=Jrb.df(a);if(a.T!=h)return true;for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],2*a.c+4,H3(v3(a.e[b]),a.c+4));}}}j=tX(Drb.$Y,mub,151,a.Q.g,0,1);for(g=0;g<j.length;g++)j[g]=new Jrb.Ag(a.e[Jrb.Ik(a.Q,0,g)],a.e[Jrb.Ik(a.Q,1,g)],g);Krb.Ueb(j,0,j.length,null);for(f=0;f<j.length;f++){if(Jrb.Ne(a,j[f].a,2)){while(f+1<j.length&&Jrb.zg(j[f],j[f+1])==0)Jrb.Ne(a,j[++f].a,2);h=a.T;a.T=Jrb.df(a);if(a.T!=h)return true;for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],2*a.c+4,H3(v3(a.e[b]),a.c+4));}}}return false;};Jrb.bf=function bf(a){var b,c,d,e,f,g,h,i,j,k;f=0;for(c=0;c<a.Q.f;c++)a.ab[c]!=0&&(a.ab[c]!=2||(a.P&256)==0)&&++f;if(f==0)return;k=tX(Drb.NY,Qsb,6,f,15,1);f=0;for(d=0;d<a.Q.f;d++){if(a.ab[d]!=0&&(a.ab[d]!=2||(a.P&256)==0)){k[f]=a.ab[d]<<29|a._[d]<<24|a.e[d]<<12|d;++f;}}Hrb.Nmb(k,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));g=0;j=0;h=k[0]&cub;while(true){++j;if(j==k.length||h!=(k[j]&cub)){e=tX(Drb.NY,Qsb,6,j-g,15,1);for(i=g;i<j;i++){b=k[i]&4095;e[i-g]=b;a.fb[b]=true;}Krb.Ei(a.gb,e);if(j==k.length)break;h=k[j]&cub;g=j;}}};Jrb.cf=function cf(a){var b,c,d,e,f,g,h,i,j,k,l,m,n;h=false;for(i=0;i<a.gb.a.length;i++){g=Krb.Ji(a.gb,i);b=true;n=-1;j=false;for(l=0;l<g.length;l++){c=g[l];if(a.cb[c]==0){b=false;break;}if(a.cb[c]!=3){k=true;for(m=0;m<g.length;m++){if(m!=l&&a.e[c]==a.e[g[m]]){k=false;break;}}if(k&&n<a.e[c]){n=a.e[c];j=a.cb[c]==1;}}}if(b&&n!=-1){for(d=g,e=0,f=d.length;e<f;++e){c=d[e];(a.cb[c]==1||a.cb[c]==2)&&(a.db[c]=j);a.fb[c]=false;}Krb.Ri(a.gb,g);h=true;--i;}}return h;};Jrb.df=function df(a){var b,c;b=Jrb.Ve(a);do{c=b;Jrb.Re(a);b=Jrb.Ve(a);}while(c!=b);return b;};Jrb.ef=function ef(a){(a.P&1)!=0&&(a.P&2)==0&&(a.f=Krb.Ceb(a.e,a.Q.f));if((a.P&Vtb)==0){a.V=tX(Drb.c3,itb,6,a.Q.f,16,1);a.U=tX(Drb.c3,itb,6,a.Q.g,16,1);if(a.T<a.Q.f){Jrb.Fe(a);if((a.P&Vtb)==0){Jrb.cf(a);!!a.O&&Jrb.Mg(a.O,a.e);}}}a.f==null&&(a.P&1)!=0&&(a.P&2)!=0&&(a.f=Krb.Ceb(a.e,a.Q.f));while(a.T<a.Q.f){Jrb.Ge(a);if((a.P&Vtb)==0){Jrb.cf(a);!!a.O&&Jrb.Mg(a.O,a.e);}}if((a.P&Vtb)==0){Jrb.cf(a);Jrb.Ze(a);Jrb.Df(a);}};Jrb.ff=function ff(a){var b,c,d,e,f;f=a.T;e=Krb.Ceb(a.e,a.Q.f);if(!a.Q.K){Jrb.hf(a);Jrb.Vf(a,f,e);}a.ab=tX(Drb.JY,nub,6,a.Q.f,15,1);a._=tX(Drb.JY,nub,6,a.Q.f,15,1);for(c=0;c<a.Q.f;c++){a.ab[c]=Jrb.rk(a.Q,c)<<24>>24;a._[c]=Jrb.qk(a.Q,c)<<24>>24;}a.k=tX(Drb.JY,nub,6,a.Q.g,15,1);a.j=tX(Drb.JY,nub,6,a.Q.g,15,1);for(d=0;d<a.Q.g;d++){a.k[d]=Jrb.Nk(a.Q,d)<<24>>24;a.j[d]=Jrb.Mk(a.Q,d)<<24>>24;}Jrb.gf(a);a.Y=false;a.L=tX(Drb.c3,itb,6,a.Q.f,16,1);for(b=0;b<a.Q.f;b++){if(a.cb[b]!=0){a.L[b]=true;a.Y=true;}}Jrb.kf(a);a.O=null;a.bb=tX(Drb.c3,itb,6,a.Q.f,16,1);if(a.Y){a.O=new Jrb.Sg(a.Q,e,a.L,a.cb,a.n,a.ab,a._,a.hb,a.p,a.bb);Jrb.Ng(a.O);}a.db=tX(Drb.c3,itb,6,a.Q.f,16,1);a.fb=tX(Drb.c3,itb,6,a.Q.f,16,1);a.gb=new Krb.Yi();!!a.O&&Jrb.Mg(a.O,a.e);Jrb.bf(a);Jrb.Vf(a,f,e);Jrb.jf(a);!!a.O&&(a.J=Jrb.Jg(a.O));Jrb.uf(a);};Jrb.gf=function gf(a){var b,c,d,e,f,g;a.K=true;f=Jrb.Ye(a,false);while(a.T<a.Q.f&&f){for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));g=a.cb[b]<<7;if((a.cb[b]==1||a.cb[b]==2)&&a.ab[b]!=0){g|=a.ab[b]<<5;g|=a._[b];}Jrb.tg(a.d[b],18,H3(Drb.r3(g),9));}for(c=0;c<a.Q.g;c++){d=a.n[c]<<7;if((a.n[c]==1||a.n[c]==2)&&Jrb.Tk(a.Q,c)==1&&a.k[c]!=0){d|=a.k[c]<<5;d|=a.j[c];}Jrb.ug(a.d[Jrb.Ik(a.Q,0,c)],Drb.r3(d));Jrb.ug(a.d[Jrb.Ik(a.Q,1,c)],Drb.r3(d));}e=Jrb.df(a);if(a.T==e)break;a.T=e;f=Jrb.Ye(a,false);}};Jrb.hf=function hf(a){var b,c,d,e;a.K=true;a.Z=tX(Drb.JY,nub,6,a.Q.f,15,1);a.g=tX(Drb.JY,nub,6,a.Q.g,15,1);e=Jrb.Ye(a,true);while(a.T<a.Q.f&&e){for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c+4,G3(H3(v3(a.e[b]),4),v3(a.cb[b]<<2)));}for(c=0;c<a.Q.g;c++){Jrb.ug(a.d[Jrb.Ik(a.Q,0,c)],v3(a.n[c]));Jrb.ug(a.d[Jrb.Ik(a.Q,1,c)],v3(a.n[c]));}d=Jrb.df(a);if(a.T==d)break;a.T=d;e=Jrb.Ye(a,true);}};Jrb.jf=function jf(a){var b,c,d,e,f,g,h;a.K=true;d=Jrb.tf(a);!!a.O&&Jrb.Mg(a.O,a.e)&&(d=Jrb.tf(a));Jrb.Ye(a,false)&&Jrb.cf(a);g=true;while(a.T<a.Q.f&&g){e=Jrb.$e(a,d);for(b=0;b<a.Q.f;b++){Jrb.wg(a.d[b],b);Jrb.tg(a.d[b],a.c,v3(a.e[b]));Jrb.tg(a.d[b],20,0);!a.bb[b]&&a.ab[b]!=0&&Jrb.ug(a.d[b],i3(v3(a.ab[b]<<18),H3(v3(e[a.ab[b]==1?0:1][a._[b]]),8)));h=a.cb[b];a.db[b]&&(h==1?h=2:h==2&&(h=1));Jrb.ug(a.d[b],Drb.r3(h<<4));}for(c=0;c<a.Q.g;c++){Jrb.ug(a.d[Jrb.Ik(a.Q,0,c)],v3(a.n[c]));Jrb.ug(a.d[Jrb.Ik(a.Q,1,c)],v3(a.n[c]));}f=Jrb.df(a);if(a.T==f)break;a.T=f;g=false;if(!!a.O&&Jrb.Mg(a.O,a.e)){g=true;d=Jrb.tf(a);}if(Jrb.Ye(a,false)){g=true;Jrb.cf(a);}}};Jrb.kf=function kf(a){var b,c;for(b=0;b<a.Q.f;b++)(!a.L[b]||a.cb[b]==3)&&(a.ab[b]=0);for(c=0;c<a.Q.g;c++)(Jrb.Tk(a.Q,c)!=1||a.n[c]==0||a.n[c]==3)&&(a.k[c]=0);};Jrb.lf=function lf(a,b,c){if(a.a==null){a.a=tX(Drb.JY,nub,6,a.Q.f,15,1);Krb.Feb(a.a);}a.a[b]=c<<24>>24;};Jrb.mf=function mf(b,c){var d,e,f,g,h,i;if((b.n[c]==1||b.n[c]==2)&&!Jrb.Po(b.Q,c)){h=false;try{for(g=0;g<2;g++){d=Jrb.Ik(b.Q,g,c);if(Jrb.Gn(b.Q,d)==3){e=tX(Drb.NY,Qsb,6,2,15,1);f=0;for(i=0;i<Jrb.Gn(b.Q,d);i++)Jrb.Hn(b.Q,d,i)!=c&&(e[f++]=Jrb.Fn(b.Q,d,i));b.e[e[0]]>b.e[e[1]]^Jrb.of(b,d,e[0],e[1])&&(h=!h);}}}catch(a){a=g3(a);if(zY(a,15)){b.g[c]=3;return;}else throw h3(a);}b.n[c]==1^h?b.g[c]=1:b.g[c]=2;}};Jrb.nf=function nf(b,c){var d,e,f,g,h,i,j;if(b.cb[c]==1||b.cb[c]==2){i=false;if(Jrb.wn(b.Q,c)==2&&Jrb.Gn(b.Q,c)==2){try{for(h=0;h<2;h++){d=Jrb.Fn(b.Q,c,h);if(Jrb.Gn(b.Q,d)==3){f=tX(Drb.NY,Qsb,6,2,15,1);g=0;for(j=0;j<Jrb.Gn(b.Q,d);j++)Jrb.In(b.Q,d,j)==1&&(f[g++]=Jrb.Fn(b.Q,d,j));b.e[f[0]]>b.e[f[1]]^Jrb.of(b,d,f[0],f[1])&&(i=!i);}}}catch(a){a=g3(a);if(zY(a,15)){b.Z[c]=3;return;}else throw h3(a);}}else {try{e=Jrb.qf(b,c);}catch(a){a=g3(a);if(zY(a,15)){b.Z[c]=3;return;}else throw h3(a);}for(h=1;h<e.length;h++)for(j=0;j<h;j++)b.e[e[h]]<b.e[e[j]]&&(i=!i);}b.cb[c]==1^i?b.Z[c]=1:b.Z[c]=2;}};Jrb.of=function of(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L;if(Jrb.Dk(a.Q,c)!=Jrb.Dk(a.Q,d))return Jrb.Dk(a.Q,c)>Jrb.Dk(a.Q,d);if(Jrb.wk(a.Q,c)!=Jrb.wk(a.Q,d)){H=Jrb.ul(a.Q,c)?(Jrb.Ej(),Jrb.Cj)[Jrb.Dk(a.Q,c)]:Jrb.wk(a.Q,c);I=Jrb.ul(a.Q,d)?(Jrb.Ej(),Jrb.Cj)[Jrb.Dk(a.Q,d)]:Jrb.wk(a.Q,d);return H>I;}w=a.Q.f;s=tX(Drb.NY,Qsb,6,w,15,1);u=tX(Drb.NY,Qsb,6,w,15,1);v=tX(Drb.NY,Qsb,6,w,15,1);t=tX(Drb.c3,itb,6,w,16,1);i=tX(Drb.c3,itb,6,a.Q.q,16,1);s[0]=b;s[1]=c;s[2]=d;u[0]=-1;u[1]=0;u[2]=0;i[b]=true;i[c]=true;i[d]=true;m=1;A=2;G=tX(Drb.NY,Qsb,6,64,15,1);G[1]=1;G[2]=3;o=2;while(m<=A){while(m<G[o]){n=s[m];if(!t[m]){p=0;q=0;for(C=0;C<Jrb.Gn(a.Q,n);C++){k=Jrb.Fn(a.Q,n,C);if(A+Jrb.In(a.Q,n,C)+1>=w){w+=a.Q.f;s=Jrb.Zf(s,w);u=Jrb.Zf(u,w);v=Jrb.Zf(v,w);t=Jrb.$f(t,w);}if(Jrb.Do(a.Q,Jrb.Hn(a.Q,n,C))){++p;q+=Jrb.Dk(a.Q,k);}else if(k!=b){for(F=1;F<Jrb.In(a.Q,n,C);F++){++A;s[A]=k;u[A]=m;t[A]=true;}}K=u[m];if(k==s[K])continue;h=false;if(i[k]){J=u[K];while(J!=-1){if(k==s[J]){h=true;break;}J=u[J];}}if(h){++A;s[A]=k;u[A]=m;t[A]=true;}else {++A;s[A]=k;u[A]=m;i[k]=true;}}if(p!=0){++A;v[A]=(q<<2)/p|0;u[A]=m;t[A]=true;}}++m;if(m==Hsb){throw h3(new Erb._z('Emergency break in while loop.'));}}G.length==o+1&&(G=Jrb.Zf(G,G.length+64));G[o+1]=A+1;for(B=G[o];B<G[o+1];B++){v[B]==0&&(v[B]=(Jrb.Dk(a.Q,s[B])==151?1:Jrb.Dk(a.Q,s[B])==152?1:Jrb.Dk(a.Q,s[B]))<<2);v[B]+=v[u[B]]<<16;}Jrb.sf(a,t,v,u,s,G,o);if(v[1]!=v[2])return v[1]>v[2];o>1&&Jrb.pf(v,u,G,o);++o;}l=tX(Drb.NY,Qsb,6,a.Q.f,15,1);D=false;for(f=0;f<a.Q.f;f++){if(i[f]&&!Jrb.ul(a.Q,f)){D=true;break;}}if(D){for(g=0;g<a.Q.f;g++)l[g]=Jrb.ul(a.Q,g)?(Jrb.Ej(),Jrb.Cj)[Jrb.Dk(a.Q,g)]:Jrb.wk(a.Q,g);if(Jrb.rf(a,t,v,u,s,l,G,o))return v[1]>v[2];}Krb.Meb(l,l.length,0);r=false;for(j=0;j<a.Q.g;j++){if(i[Jrb.Ik(a.Q,0,j)]||i[Jrb.Ik(a.Q,1,j)]){if(a.g[j]==1){l[Jrb.Ik(a.Q,0,j)]=1;l[Jrb.Ik(a.Q,1,j)]=1;r=true;}else if(a.g[j]==2){l[Jrb.Ik(a.Q,0,j)]=2;l[Jrb.Ik(a.Q,1,j)]=2;r=true;}}}if(r&&Jrb.rf(a,t,v,u,s,l,G,o))return v[1]>v[2];Krb.Meb(l,l.length,0);L=false;for(e=0;e<a.Q.f;e++){if(i[e]){if(a.Z[e]==2){l[e]=1;L=true;}else if(a.Z[e]==1){l[e]=2;L=true;}}}if(L&&Jrb.rf(a,t,v,u,s,l,G,o))return v[1]>v[2];throw h3(new Erb._z('no distinction applying CIP rules'));};Jrb.pf=function pf(a,b,c,d){var e,f,g,h,i,j,k,l,m;l=c[d];g=c[d+1]-l;m=tX(Drb.XY,Bsb,153,g,0,1);for(i=0;i<g;i++){m[i]=new Jrb.sg();m[i].c=a[i+l];m[i].b=b[i+l];m[i].a=i+l;}e=new Jrb.pg();for(k=d;k>1;k--){for(j=0;j<g;j++){m[j].c+=a[m[j].b]<<16;m[j].b=b[m[j].b];}Krb.Ueb(m,0,m.length,e);f=1;for(h=0;h<g;h++){a[m[h].a]=f;h!=g-1&&Jrb.og(m[h],m[h+1])!=0&&++f;}}};Jrb.qf=function qf(a,b){var c,d,e,f,g,h,i;g=Jrb.rn(a.Q,b);h=tX(Drb.NY,Qsb,6,g,15,1);for(e=0;e<g;e++)h[e]=Jrb.Fn(a.Q,b,e);for(d=g;d>1;d--){c=false;for(f=1;f<d;f++){if(Jrb.of(a,b,h[f-1],h[f])){c=true;i=h[f-1];h[f-1]=h[f];h[f]=i;}}if(!c)break;}return h;};Jrb.rf=function rf(a,b,c,d,e,f,g,h){var i,j;for(j=1;j<h;j++){for(i=g[j];i<g[j+1];i++)c[i]=f[e[i]]+(c[d[i]]<<8);Jrb.sf(a,b,c,d,e,g,j);if(c[1]!=c[2])return true;j>1&&Jrb.pf(c,d,g,j);}return false;};Jrb.sf=function sf(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p,q,r;for(l=g;l>1;l--){p=f[l]-f[l-1];r=tX(Drb.VY,Bsb,152,p,0,1);h=f[l];for(o=0;o<p;o++){q=f[l-1]+o;m=h;while(m<f[l+1]&&d[m]==q)++m;r[o]=new Jrb.mg();r[o].c=q;r[o].d=c[q];r[o].b=b[q]?0:Jrb.ho(a.Q,e[q]);r[o].a=tX(Drb.NY,Qsb,6,m-h,15,1);for(k=h;k<m;k++)r[o].a[k-h]=c[k];Hrb.Nmb(r[o].a,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));h=m;}i=new Jrb.jg();Krb.Ueb(r,0,r.length,i);j=1;for(n=0;n<p;n++){c[r[n].c]=j;n!=p-1&&Jrb.ig(r[n],r[n+1])!=0&&++j;}}};Jrb.tf=function tf(a){var b,c;c=rX(Drb.NY,[ssb,lub],[25,7],0,[2,32],2);for(b=0;b<a.Q.f;b++){a.L[b]&&(a.ab[b]==1?c[0][a._[b]]=Jrb.Tg(c[0][a._[b]],b):a.ab[b]==2&&(c[1][a._[b]]=Jrb.Tg(c[0][a._[b]],b)));}return c;};Jrb.uf=function uf(a){var b,c,d,e,f,g,h,i,j,k,l,m;f=0;k=0;g=0;h=0;i=0;j=0;l=0;m=false;b=tX(Drb.c3,itb,6,32,16,1);for(c=0;c<a.Q.f;c++){if(a.cb[c]!=0){++f;if(a.cb[c]==3){++k;}else {if(a.ab[c]==0){++g;!!a.O&&Jrb.Ig(a.O,c)&&++h;}else if(a.ab[c]==2){a._[c]==0&&++j;}else if(a.ab[c]==1){e=a._[c];if(!b[e]){++l;b[e]=true;}a._[c]==0&&++i;!!a.O&&Jrb.Ig(a.O,c)&&(m=true);}}}}for(d=0;d<a.Q.g;d++){if(a.n[d]!=0&&Jrb.Tk(a.Q,d)==1){++f;if(a.n[d]==3){++k;}else {if(a.k[d]==0){++g;!!a.O&&Jrb.Ig(a.O,Jrb.Ik(a.Q,0,d))&&Jrb.Ig(a.O,Jrb.Ik(a.Q,1,d))&&++h;}else if(a.k[d]==2){a.j[d]==0&&++j;}else if(a.k[d]==1){e=a.j[d];if(!b[e]){++l;b[e]=true;}a.j[d]==0&&++i;!!a.O&&Jrb.Ig(a.O,Jrb.Ik(a.Q,0,d))&&Jrb.Ig(a.O,Jrb.Ik(a.Q,1,d))&&(m=true);}}}}if(f==0){Jrb.um(a.Q,Fsb);return;}if(k!=0){Jrb.um(a.Q,0);return;}if(a.J){Jrb.um(a.Q,oub+(1<<l));return;}i+h==f&&!m?Jrb.um(a.Q,196608):g==f?Jrb.um(a.Q,bub):j==f?Jrb.um(a.Q,327680):g==f-1&&i==1?Jrb.um(a.Q,Ftb):Jrb.um(a.Q,458752+(1<<l));};Jrb.vf=function vf(a,b,c){while(c!=0){if(a.s==0){(!a.q||a.u!=63)&&(a.u+=64);Erb.L7(a.t,a.u&Gsb);a.s=6;a.u=0;}a.u<<=1;a.u=N3(G3(v3(a.u),j3(b,1)));b=I3(b,1);--c;--a.s;}};Jrb.wf=function wf(a){a.u<<=a.s;(!a.q||a.u!=63)&&(a.u+=64);Erb.L7(a.t,a.u&Gsb);return a.t.a;};Jrb.xf=function xf(a,b){a.t=new Erb.S7();a.s=6;a.u=0;a.q=b;};Jrb.yf=function yf(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q;if(a.Q.f==0){a.r='';return;}l=a.ib&a.b;q=a.ib?16:8;Jrb.xf(a,true);Erb.L7(a.t,l?35:33);Jrb.vf(a,v3(a.ib?1:0),1);Jrb.vf(a,Drb.r3(b?1:0),1);Jrb.vf(a,Drb.r3(q>>1),4);o=0;for(i=1;i<a.Q.f;i++)o=Jrb.Lf(a,a.A[i],a.D[i]==-1?-1:a.A[a.D[i]],o,c);if(l){for(j=0;j<a.Q.f;j++){d=a.A[j];for(n=Jrb.Gn(a.Q,d);n<Jrb.rn(a.Q,d);n++)o=Jrb.Lf(a,Jrb.Fn(a.Q,d,n),d,o,c);}}if(a.Q.f>1&&o==0){a.r='';return;}g=1<<q;m=o/((g>>1)-1);p=o+0.5*m;for(k=1;k<a.Q.f;k++)Jrb.zf(a,a.A[k],a.D[k]==-1?-1:a.A[a.D[k]],p,m,q,c);if(l){for(h=0;h<a.Q.f;h++){d=a.A[h];for(n=Jrb.Gn(a.Q,d);n<Jrb.rn(a.Q,d);n++)Jrb.zf(a,Jrb.Fn(a.Q,d,n),d,p,m,q,c);}}if(b){f=a.ib?1.5:(Jrb.Ej(),Jrb.Ej(),Jrb.Dj);e=Jrb.Gk(a.Q,l?a.Q.q:a.Q.f,l?a.Q.r:a.Q.g,f,c);Jrb.vf(a,v3($wnd.Math.min(g-1,$wnd.Math.max(0,IY(0.5+$wnd.Math.log(e/0.1)*$wnd.Math.LOG10E/($wnd.Math.log(2000)*$wnd.Math.LOG10E)*(g-1))))),q);Jrb.vf(a,v3(Jrb.Cf(c[a.A[0]].a/e,g)),q);Jrb.vf(a,v3(Jrb.Cf(c[a.A[0]].b/e,g)),q);a.ib&&Jrb.vf(a,v3(Jrb.Cf(c[a.A[0]].c/e,g)),q);}a.r=Jrb.wf(a);};Jrb.zf=function zf(a,b,c,d,e,f,g){var h,i,j;h=c==-1?(g[b].a-g[a.A[0]].a)/8:g[b].a-g[c].a;i=c==-1?(g[b].b-g[a.A[0]].b)/8:g[b].b-g[c].b;Jrb.vf(a,v3(IY((d+h)/e)),f);Jrb.vf(a,v3(IY((d+i)/e)),f);if(a.ib){j=c==-1?(g[b].c-g[a.A[0]].c)/8:g[b].c-g[c].c;Jrb.vf(a,v3(IY((d+j)/e)),f);}};Jrb.Af=function Af(a,b){var c;for(c=0;c<a.v;c++)b-=16;b<0&&(Erb.X7(),String.fromCharCode(10));while(b>15){Jrb.vf(a,1,1);Jrb.vf(a,15,4);b-=16;++a.v;}Jrb.vf(a,1,1);Jrb.vf(a,Drb.r3(b),4);};Jrb.Bf=function Bf(a){var b,c,d,e,f,g;if(a.Q.f==0){a.M='';return;}f=0;d=false;e=false;for(c=0;c<a.Q.f;c++){f<Jrb.vk(a.Q,c)&&(f=Jrb.vk(a.Q,c));Jrb.jl(a.Q,c)?d=true:e=true;}if(f==0){a.M='';return;}g=Jrb.gg(f);Jrb.xf(a,true);Jrb.vf(a,Drb.r3(g),4);Jrb.vf(a,Drb.r3(d?1:0),1);Jrb.vf(a,Drb.r3(e?1:0),1);for(b=0;b<a.Q.f;b++){Jrb.vf(a,v3(Jrb.vk(a.Q,a.A[b])),g);d&&e&&Jrb.vf(a,v3(Jrb.jl(a.Q,a.A[b])?1:0),1);}a.M=Jrb.wf(a);};Jrb.Cf=function Cf(a,b){var c,d,e,f;c=b/2|0;e=a<0;a=$wnd.Math.abs(a);f=b/32|0;d=$wnd.Math.min(c-1,N3(u3($wnd.Math.round(a*c/(a+f)))));return e?c+d:d;};Jrb.Df=function Df(a){var b,c,d;for(b=0;b<a.Q.f;b++){a.cb[b]==3&&!Jrb.fl(a.Q,b)&&Jrb.Cm(a.Q,b);(Jrb.rk(a.Q,b)==1||Jrb.rk(a.Q,b)==2)&&a.cb[b]==3&&Jrb.Cm(a.Q,b);Jrb.fl(a.Q,b)&&a.cb[b]!=3&&!Jrb.Xf(a,b)&&Jrb.Cm(a.Q,b);}for(d=0;d<a.Q.r;d++)Jrb.zl(a.Q,d)&&!Jrb.Wf(a,d)&&Jrb.Cm(a.Q,Jrb.Ik(a.Q,0,d));for(c=0;c<a.Q.g;c++){if(Jrb.Qk(a.Q,c)==2){if(Jrb.pl(a.Q,c)&&(a.n[c]==1||a.n[c]==2)){a.n[c]=3;Jrb.tm(a.Q,c,386);}if(a.n[c]==3&&!a.o[c]){if(Jrb.Tk(a.Q,c)!=386){Jrb.Cm(a.Q,Jrb.Ik(a.Q,0,c));Jrb.Cm(a.Q,Jrb.Ik(a.Q,1,c));}}}if(Jrb.Tk(a.Q,c)==1&&a.n[c]==3&&!Jrb.fl(a.Q,Jrb.Ik(a.Q,0,c))&&!Jrb.fl(a.Q,Jrb.Ik(a.Q,1,c))){Jrb.Cm(a.Q,Jrb.Ik(a.Q,0,c));Jrb.Cm(a.Q,Jrb.Ik(a.Q,1,c));}if((Jrb.Nk(a.Q,c)==1||Jrb.Nk(a.Q,c)==2)&&(Jrb.Tk(a.Q,c)!=1||a.n[c]!=1&&a.n[c]!=2)){Jrb.Cm(a.Q,Jrb.Ik(a.Q,0,c));Jrb.Cm(a.Q,Jrb.Ik(a.Q,1,c));}}};Jrb.Ef=function Ef(a){var b,c,d,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v;if(a.Q.f==0)return;if(a.F)return;a.H=0;v=0;for(c=1;c<a.Q.f;c++)a.e[c]>a.e[v]&&(v=c);d=tX(Drb.c3,itb,6,a.Q.f,16,1);g=tX(Drb.c3,itb,6,a.Q.g,16,1);a.G=tX(Drb.NY,Qsb,6,a.Q.f,15,1);a.A=tX(Drb.NY,Qsb,6,a.Q.f,15,1);a.D=tX(Drb.NY,Qsb,6,a.Q.f,15,1);a.B=tX(Drb.NY,Qsb,6,a.Q.g,15,1);a.A[0]=v;a.G[v]=0;d[v]=true;i=0;j=1;k=0;while(i<a.Q.f){if(i<j){while(true){o=0;p=0;m=-1;b=a.A[i];for(q=0;q<Jrb.sn(a.Q,b);q++){if(q<Jrb.Gn(a.Q,b)||q>=Jrb.rn(a.Q,b)){h=Jrb.Fn(a.Q,b,q);if(!d[h]&&a.e[h]>m){o=h;p=Jrb.Hn(a.Q,b,q);m=a.e[h];}}}if(m==-1)break;a.G[o]=j;a.D[j]=i;a.A[j++]=o;a.B[k++]=p;d[o]=true;g[p]=true;}++i;}else {n=0;m=-1;for(b=0;b<a.Q.f;b++){if(!d[b]&&a.e[b]>m){n=b;m=a.e[b];}}a.G[n]=j;a.D[j]=-1;a.A[j++]=n;d[n]=true;}}a.C=tX(Drb.NY,Qsb,6,2*(a.Q.g-k),15,1);while(true){s=a.Q.M;t=a.Q.M;u=-1;for(f=0;f<a.Q.g;f++){if(!g[f]){if(a.G[Jrb.Ik(a.Q,0,f)]<a.G[Jrb.Ik(a.Q,1,f)]){r=a.G[Jrb.Ik(a.Q,0,f)];l=a.G[Jrb.Ik(a.Q,1,f)];}else {r=a.G[Jrb.Ik(a.Q,1,f)];l=a.G[Jrb.Ik(a.Q,0,f)];}if(r<s||r==s&&l<t){s=r;t=l;u=f;}}}if(u==-1)break;g[u]=true;a.B[k++]=u;a.C[2*a.H]=s;a.C[2*a.H+1]=t;++a.H;}a.F=true;};Jrb.Ff=function Ff(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o;i=null;o=Jrb.jo(a.Q);for(l=0;l<o.i.a.length;l++){if(o.e[l]){f=0;m=Krb.Ji(o.i,l);for(c=m,d=0,e=c.length;d<e;++d){b=c[d];Jrb.Rf(a,b)&&++f;}if(f!=0){n=Krb.Ji(o.j,l);i==null&&(i=tX(Drb.c3,itb,6,a.Q.g,16,1));if(f==m.length){j=-1;k=xsb;for(g=0;g<m.length;g++){if(k>a.A[n[g]]){k=a.A[n[g]];j=g;}}while(f>0){i[n[j]]=true;j=Jrb.dg(j+2,m.length);f-=2;}}else {h=0;while(Jrb.Rf(a,m[h]))++h;while(!Jrb.Rf(a,m[h]))h=Jrb.dg(h+1,m.length);while(f>0){i[n[h]]=true;h=Jrb.dg(h+2,m.length);f-=2;while(!Jrb.Rf(a,m[h]))h=Jrb.dg(h+1,m.length);}}}}}return i;};Jrb.Gf=function Gf(a,b){return a.n[b];};Jrb.Hf=function Hf(a){return Jrb.If(a,a.ib);};Jrb.If=function If(a,b){if(a.r==null){Jrb.Ef(a);Jrb.yf(a,b,a.Q.J);}return a.r;};Jrb.Jf=function Jf(a){if(a.M==null){Jrb.Ef(a);Jrb.Bf(a);}return a.M;};Jrb.Kf=function Kf(a){if(a.I==null){Jrb.Ef(a);if((a.P&Vtb)==0){Jrb.Tf(a);Jrb.Uf(a,1);Jrb.Uf(a,2);}Jrb.Sf(a);}return a.I;};Jrb.Lf=function Lf(a,b,c,d,e){var f,g,h;f=c==-1?$wnd.Math.abs(e[b].a-e[a.A[0]].a)/8:$wnd.Math.abs(e[b].a-e[c].a);d<f&&(d=f);g=c==-1?$wnd.Math.abs(e[b].b-e[a.A[0]].b)/8:$wnd.Math.abs(e[b].b-e[c].b);d<g&&(d=g);if(a.ib){h=c==-1?$wnd.Math.abs(e[b].c-e[a.A[0]].c)/8:$wnd.Math.abs(e[b].c-e[c].c);d<h&&(d=h);}return d;};Jrb.Mf=function Mf(a,b){return a.W[b];};Jrb.Nf=function Nf(a,b){return a.X[b];};Jrb.Of=function Of(a,b){return a.f==null?-1:a.f[b];};Jrb.Pf=function Pf(a,b){return a.cb[b];};Jrb.Qf=function Qf(a,b){var c,d,e,f,g,h,i;i=Jrb.jo(a.Q);for(c=0;c<i.i.a.length;c++){if(i.d[c]&&Jrb.Hr(i,c,b)){for(e=Krb.Ji(i.i,c),f=0,g=e.length;f<g;++f){d=e[f];if(d!=b)for(h=0;h<Jrb.Gn(a.Q,d);h++)if(Jrb.Ao(a.Q,Jrb.Hn(a.Q,d,h)))return true;}return false;}}return false;};Jrb.Rf=function Rf(a,b){var c,d,e;if(Jrb.wn(a.Q,b)<2)return false;if(Jrb.Gn(a.Q,b)==2)return true;c=0;for(e=0;e<Jrb.Gn(a.Q,b);e++){d=Jrb.Hn(a.Q,b,e);Jrb.zo(a.Q,d)&&(c+=Jrb.Qk(a.Q,d)-1);}return c>1;};Jrb.Sf=function Sf(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$,ab,bb;Jrb.xf(a,false);Jrb.vf(a,9,4);Y=$wnd.Math.max(Jrb.gg(a.Q.f),Jrb.gg(a.Q.g));Jrb.vf(a,Drb.r3(Y),4);if(Y==0){Jrb.vf(a,v3(a.Q.K?1:0),1);Jrb.vf(a,0,1);a.I=Jrb.wf(a);return;}Z=ab=$=H=0;for(i=0;i<a.Q.f;i++){if(t3(j3(Jrb.yk(a.Q,i),1),0)){switch(Jrb.Dk(a.Q,i)){case 6:break;case 7:++Z;break;case 8:++ab;break;default:++$;}Jrb.mk(a.Q,i)!=0&&++H;}}Jrb.vf(a,v3(a.Q.f),Y);Jrb.vf(a,v3(a.Q.g),Y);Jrb.vf(a,Drb.r3(Z),Y);Jrb.vf(a,Drb.r3(ab),Y);Jrb.vf(a,Drb.r3($),Y);Jrb.vf(a,Drb.r3(H),Y);for(j=0;j<a.Q.f;j++)Jrb.Dk(a.Q,a.A[j])==7&&t3(j3(Jrb.yk(a.Q,a.A[j]),1),0)&&Jrb.vf(a,Drb.r3(j),Y);for(m=0;m<a.Q.f;m++)Jrb.Dk(a.Q,a.A[m])==8&&t3(j3(Jrb.yk(a.Q,a.A[m]),1),0)&&Jrb.vf(a,Drb.r3(m),Y);for(n=0;n<a.Q.f;n++)if(Jrb.Dk(a.Q,a.A[n])!=6&&Jrb.Dk(a.Q,a.A[n])!=7&&Jrb.Dk(a.Q,a.A[n])!=8&&t3(j3(Jrb.yk(a.Q,a.A[n]),1),0)){Jrb.vf(a,Drb.r3(n),Y);Jrb.vf(a,v3(Jrb.Dk(a.Q,a.A[n])),8);}for(o=0;o<a.Q.f;o++)if(Jrb.mk(a.Q,a.A[o])!=0&&t3(j3(Jrb.yk(a.Q,a.A[o]),1),0)){Jrb.vf(a,Drb.r3(o),Y);Jrb.vf(a,v3(8+Jrb.mk(a.Q,a.A[o])),4);}X=0;v=0;for(p=1;p<a.Q.f;p++){if(a.D[p]==-1){N=0;}else {N=1+a.D[p]-v;v=a.D[p];}X<N&&(X=N);}M=Jrb.gg(X);Jrb.vf(a,Drb.r3(M),4);v=0;for(q=1;q<a.Q.f;q++){if(a.D[q]==-1){N=0;}else {N=1+a.D[q]-v;v=a.D[q];}Jrb.vf(a,Drb.r3(N),M);}for(Q=0;Q<2*a.H;Q++)Jrb.vf(a,v3(a.C[Q]),Y);for(A=0;A<a.Q.g;A++){G=(Jrb.Sk(a.Q,a.B[A])&Xtb)!=0||Jrb.Tk(a.Q,a.B[A])==16?1:Jrb.Do(a.Q,a.B[A])?0:$wnd.Math.min(3,Jrb.Qk(a.Q,a.B[A]));Jrb.vf(a,Drb.r3(G),2);}c=0;if((a.P&Vtb)==0){for(r=0;r<a.Q.f;r++)a.$[a.A[r]]!=0&&a.$[a.A[r]]!=3&&++c;}Jrb.vf(a,Drb.r3(c),Y);if((a.P&Vtb)==0){for(r=0;r<a.Q.f;r++){if(a.$[a.A[r]]!=0&&a.$[a.A[r]]!=3){Jrb.vf(a,Drb.r3(r),Y);if(a.ab[a.A[r]]==0){Jrb.vf(a,v3(a.$[a.A[r]]),3);}else {bb=a.$[a.A[r]]==1?a.ab[a.A[r]]==1?4:6:a.ab[a.A[r]]==1?5:7;Jrb.vf(a,Drb.r3(bb),3);Jrb.vf(a,v3(a._[a.A[r]]),3);}}}}b=0;if((a.P&Vtb)==0){for(B=0;B<a.Q.g;B++)a.i[a.B[B]]!=0&&a.i[a.B[B]]!=3&&(!Jrb.Po(a.Q,a.B[B])||Jrb.Tk(a.Q,a.B[B])==1)&&++b;}Jrb.vf(a,Drb.r3(b),Y);if((a.P&Vtb)==0){for(B=0;B<a.Q.g;B++){if(a.i[a.B[B]]!=0&&a.i[a.B[B]]!=3&&(!Jrb.Po(a.Q,a.B[B])||Jrb.Tk(a.Q,a.B[B])==1)){Jrb.vf(a,Drb.r3(B),Y);if(Jrb.Tk(a.Q,a.B[B])==1){if(a.k[a.B[B]]==0){Jrb.vf(a,v3(a.i[a.B[B]]),3);}else {bb=a.i[a.B[B]]==1?a.k[a.B[B]]==1?4:6:a.k[a.B[B]]==1?5:7;Jrb.vf(a,Drb.r3(bb),3);Jrb.vf(a,v3(a.j[a.B[B]]),3);}}else {Jrb.vf(a,v3(a.i[a.B[B]]),2);}}}}Jrb.vf(a,v3(a.Q.K?1:0),1);K=0;for(s=0;s<a.Q.f;s++)Jrb.wk(a.Q,a.A[s])!=0&&++K;if(K!=0){Jrb.Af(a,1);Jrb.vf(a,Drb.r3(K),Y);for(r=0;r<a.Q.f;r++){if(Jrb.wk(a.Q,a.A[r])!=0){Jrb.vf(a,Drb.r3(r),Y);Jrb.vf(a,v3(Jrb.wk(a.Q,a.A[r])),8);}}}a.v=0;if(a.Q.K){Jrb.Ce(a,0,Y,Vtb,1,-1);Jrb.Ce(a,3,Y,qtb,1,-1);Jrb.Ce(a,4,Y,120,4,3);Jrb.Ce(a,5,Y,pub,2,1);Jrb.Ce(a,6,Y,1,1,-1);Jrb.Ce(a,7,Y,utb,4,7);K=0;for(r=0;r<a.Q.f;r++)Jrb.tk(a.Q,a.A[r])!=null&&++K;if(K>0){Jrb.Af(a,8);Jrb.vf(a,Drb.r3(K),Y);for(t=0;t<a.Q.f;t++){u=Jrb.tk(a.Q,a.A[t]);if(u!=null){Jrb.vf(a,Drb.r3(t),Y);Jrb.vf(a,v3(u.length),4);for(e=u,f=0,g=e.length;f<g;++f){d=e[f];Jrb.vf(a,Drb.r3(d),8);}}}}Jrb.De(a,9,Y,384,2,7);Jrb.De(a,10,Y,31,5,0);Jrb.Ce(a,11,Y,Utb,1,-1);Jrb.De(a,12,Y,Xtb,8,9);Jrb.Ce(a,13,Y,Atb,3,14);Jrb.Ce(a,14,Y,Btb,5,17);Jrb.Ce(a,16,Y,Stb,3,22);}K=0;for(k=0;k<a.Q.f;k++)a.a!=null&&a.a[a.A[k]]!=-1&&++K;if(K!=0){Jrb.Af(a,17);Jrb.vf(a,Drb.r3(K),Y);for(r=0;r<a.Q.f;r++){if(a.a!=null&&a.a[a.A[r]]!=-1){Jrb.vf(a,Drb.r3(r),Y);Jrb.vf(a,v3(a.a[a.A[r]]),4);}}}if((a.P&8)!=0||(a.P&vtb)!=0){K=0;W=0;for(r=0;r<a.Q.f;r++){U=Jrb.ok(a.Q,a.A[r]);if(U!=null){++K;W=$wnd.Math.max(W,Hrb.rnb(U).length);}}if(K!=0){V=Jrb.gg(W);Jrb.Af(a,18);Jrb.vf(a,Drb.r3(K),Y);Jrb.vf(a,Drb.r3(V),4);for(t=0;t<a.Q.f;t++){L=Jrb.ok(a.Q,a.A[t]);if(L!=null){Jrb.vf(a,Drb.r3(t),Y);Jrb.vf(a,v3(Hrb.rnb(L).length),V);for(P=0;P<Hrb.rnb(L).length;P++)Jrb.vf(a,(Hrb.mnb(P,Hrb.rnb(L).length),v3(Hrb.rnb(L).charCodeAt(P))),7);}}}}if(a.Q.K){Jrb.Ce(a,19,Y,wtb,3,25);Jrb.De(a,20,Y,Gtb,3,17);}K=0;for(l=0;l<a.Q.f;l++)Jrb.zk(a.Q,a.A[l])!=0&&++K;if(K!=0){Jrb.Af(a,21);Jrb.vf(a,Drb.r3(K),Y);for(h=0;h<a.Q.f;h++){if(Jrb.zk(a.Q,a.A[h])!=0){Jrb.vf(a,Drb.r3(h),Y);Jrb.vf(a,v3(Jrb.zk(a.Q,a.A[h])>>4),2);}}}if(a.Q.K){Jrb.Ce(a,22,Y,Ttb,1,-1);Jrb.De(a,23,Y,qub,1,-1);Jrb.De(a,24,Y,Ytb,2,21);}if((a.P&16)!=0){for(h=0;h<a.Q.f;h++){if(Jrb.xl(a.Q,a.A[h])){Jrb.Af(a,25);for(d=0;d<a.Q.f;d++)Jrb.vf(a,v3(Jrb.xl(a.Q,a.A[d])?1:0),1);break;}}}S=Jrb.Ff(a);if(S!=null){K=0;for(B=0;B<a.Q.g;B++)S[a.B[B]]&&++K;Jrb.Af(a,26);Jrb.vf(a,Drb.r3(K),Y);for(C=0;C<a.Q.g;C++)S[a.B[C]]&&Jrb.vf(a,Drb.r3(C),Y);}a.Q.K&&Jrb.Ce(a,27,Y,jtb,1,-1);K=0;for(D=0;D<a.Q.g;D++)Jrb.Tk(a.Q,a.B[D])==16&&++K;if(K!=0){Jrb.Af(a,28);Jrb.vf(a,Drb.r3(K),Y);for(B=0;B<a.Q.g;B++)Jrb.Tk(a.Q,a.B[B])==16&&Jrb.vf(a,Drb.r3(B),Y);}if(a.Q.K){Jrb.Ce(a,29,Y,rub,2,30);Jrb.Ce(a,30,Y,Usb,7,32);Jrb.Ce(a,32,Y,tub,2,44);Jrb.Ce(a,33,Y,Itb,5,39);Jrb.Ce(a,34,Y,ttb,1,-1);Jrb.De(a,35,Y,uub,1,-1);Jrb.De(a,36,Y,96,2,5);}K=0;for(F=0;F<a.Q.g;F++)(Jrb.Tk(a.Q,a.B[F])==32||Jrb.Tk(a.Q,a.B[F])==64)&&++K;if(K!=0){Jrb.Af(a,37);Jrb.vf(a,Drb.r3(K),Y);for(w=0;w<a.Q.g;w++){if(Jrb.Tk(a.Q,a.B[w])==32||Jrb.Tk(a.Q,a.B[w])==64){Jrb.vf(a,Drb.r3(w),Y);Jrb.vf(a,v3(Jrb.Tk(a.Q,a.B[w])==32?0:1),1);}}}if(a.b&&(a.P&16)!=0){K=0;I=0;for(R=0;R<a.Q.f;R++){h=a.A[R];J=0;for(T=Jrb.Gn(a.Q,h);T<Jrb.rn(a.Q,h);T++){if(Jrb.xl(a.Q,Jrb.Fn(a.Q,h,T))){O=T-Jrb.Gn(a.Q,h);J|=1<<O;I=$wnd.Math.max(I,O+1);}}J!=0&&++K;}if(K!=0){Jrb.Af(a,38);Jrb.vf(a,Drb.r3(K),Y);Jrb.vf(a,Drb.r3(I),3);for(P=0;P<a.Q.f;P++){h=a.A[P];J=0;for(T=Jrb.Gn(a.Q,h);T<Jrb.rn(a.Q,h);T++){if(Jrb.xl(a.Q,Jrb.Fn(a.Q,h,T))){O=T-Jrb.Gn(a.Q,h);J|=1<<O;}}if(J!=0){Jrb.vf(a,Drb.r3(h),Y);Jrb.vf(a,Drb.r3(J),I);}}}}Jrb.vf(a,0,1);a.I=Jrb.wf(a);};Jrb.Tf=function Tf(a){var b,c,d,e,f,g,h,i,j,k,l;a.$=tX(Drb.JY,nub,6,a.Q.f,15,1);for(b=0;b<a.Q.f;b++){if(a.cb[b]==1||a.cb[b]==2){i=a.db[b];if(Jrb.Bo(a.Q,b)){for(h=0;h<Jrb.Gn(a.Q,b);h++){e=Jrb.Fn(a.Q,b,h);l=0;k=tX(Drb.NY,Qsb,6,3,15,1);for(j=0;j<Jrb.Gn(a.Q,e);j++){k[l]=Jrb.Fn(a.Q,e,j);k[l]!=b&&++l;}l==2&&a.e[k[0]]>a.e[k[1]]^a.G[k[0]]<a.G[k[1]]&&(i=!i);}}else {for(h=1;h<Jrb.Gn(a.Q,b);h++){for(j=0;j<h;j++){f=Jrb.Fn(a.Q,b,h);g=Jrb.Fn(a.Q,b,j);a.e[f]>a.e[g]&&(i=!i);a.G[f]<a.G[g]&&(i=!i);}}}a.$[b]=a.cb[b]==1^i?1:2;}else {a.$[b]=a.cb[b];}}a.i=tX(Drb.JY,nub,6,a.Q.g,15,1);for(c=0;c<a.Q.g;c++){if(a.n[c]==1||a.n[c]==2){i=false;for(h=0;h<2;h++){d=Jrb.Ik(a.Q,h,c);if(Jrb.Gn(a.Q,d)==3){k=tX(Drb.NY,Qsb,6,2,15,1);l=0;for(j=0;j<3;j++)Jrb.Fn(a.Q,d,j)!=Jrb.Ik(a.Q,1-h,c)&&(k[l++]=Jrb.Fn(a.Q,d,j));a.e[k[0]]>a.e[k[1]]&&(i=!i);a.G[k[0]]<a.G[k[1]]&&(i=!i);}}a.i[c]=a.n[c]==1^i?1:2;}else {a.i[c]=a.n[c];}}};Jrb.Uf=function Uf(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;i=tX(Drb.NY,Qsb,6,32,15,1);j=0;for(d=0;d<a.Q.f;d++){if((a.$[d]==1||a.$[d]==2)&&a.ab[d]==b){h=a._[d];if(i[h]<a.e[d]){i[h]==0&&++j;i[h]=a.e[d];}}}for(f=0;f<a.Q.g;f++){if((a.i[f]==1||a.i[f]==2)&&a.k[f]==b&&Jrb.Tk(a.Q,f)==1){h=a.j[f];o=$wnd.Math.max(a.e[Jrb.Ik(a.Q,0,f)],a.e[Jrb.Ik(a.Q,1,f)]);if(i[h]<o){i[h]==0&&++j;i[h]=o;}}}g=tX(Drb.JY,nub,6,32,15,1);for(k=0;k<j;k++){m=-1;n=0;for(l=0;l<32;l++){if(n<i[l]){n=i[l];m=l;}}i[m]=0;g[m]=k<<24>>24;}for(c=0;c<a.Q.f;c++)(a.$[c]==1||a.$[c]==2)&&a.ab[c]==b&&(a._[c]=g[a._[c]]);for(e=0;e<a.Q.g;e++)(a.i[e]==1||a.i[e]==2)&&a.k[e]==b&&Jrb.Tk(a.Q,e)==1&&(a.j[e]=g[a.j[e]]);};Jrb.Vf=function Vf(a,b,c){var d,e;a.T=b;for(d=0;d<a.Q.f;d++){a.e[d]=c[d];a.cb[d]=0;a.hb[d]=false;}for(e=0;e<a.Q.g;e++){a.n[e]=0;a.p[e]=false;}};Jrb.Wf=function Wf(a,b){var c,d,e;c=Jrb.Ik(a.Q,0,b);if(c>=a.Q.f)return false;if(a.cb[c]==1||a.cb[c]==2)return true;if(a.cb[c]==3)return false;d=Jrb.nn(a.Q,c);if(d!=-1)return a.n[d]==1||a.n[d]==2;for(e=0;e<Jrb.Gn(a.Q,c);e++){if(Jrb.In(a.Q,c,e)==2){if(a.cb[Jrb.Fn(a.Q,c,e)]==1||a.cb[Jrb.Fn(a.Q,c,e)]==2)return true;}}return false;};Jrb.Xf=function Xf(a,b){var c;for(c=0;c<Jrb.Gn(a.Q,b);c++)if(a.n[Jrb.Hn(a.Q,b,c)]==3&&Jrb.In(a.Q,b,c)==1)return true;return false;};Jrb.Yf=function Yf(a){var b,c,d,e,f,g,h;g=tX(Drb.NY,Qsb,6,a.T+1,15,1);for(c=0;c<a.Q.f;c++){Jrb.rk(a.Q,c)==0&&(a.cb[c]==1?++g[a.e[c]]:a.cb[c]==2&&--g[a.e[c]]);}for(e=0;e<a.Q.g;e++){if(Jrb.Qk(a.Q,e)==1&&Jrb.Nk(a.Q,e)==0){if(a.n[e]==1){++g[a.e[Jrb.Ik(a.Q,0,e)]];++g[a.e[Jrb.Ik(a.Q,1,e)]];}else if(a.n[e]==2){--g[a.e[Jrb.Ik(a.Q,0,e)]];--g[a.e[Jrb.Ik(a.Q,1,e)]];}}}for(h=1;h<=a.T;h++){if(g[h]!=0){f=g[h]<0;if(f){for(b=0;b<a.Q.f;b++){Jrb.rk(a.Q,b)==0&&(a.cb[b]==1?a.cb[b]=2:a.cb[b]==2&&(a.cb[b]=1));}for(d=0;d<a.Q.g;d++){Jrb.Qk(a.Q,d)==1&&Jrb.Nk(a.Q,d)==0&&(a.n[d]==1?a.n[d]=2:a.n[d]==2&&(a.n[d]=1));}}return f;}}return false;};Jrb.Zf=function Zf(a,b){var c;c=tX(Drb.NY,Qsb,6,b,15,1);Erb.Y7(a,0,c,0,a.length);return c;};Jrb.$f=function $f(a,b){var c;c=tX(Drb.c3,itb,6,b,16,1);Erb.Y7(a,0,c,0,a.length);return c;};Jrb._f=function _f(a){var b,c;if(a.Z!=null)for(b=0;b<a.Q.f;b++)Jrb.Rl(a.Q,b,a.Z[b]);if(a.g!=null)for(c=0;c<a.Q.g;c++)Jrb.mm(a.Q,c,a.g[c]);};Jrb.ag=function ag(a){var b,c,d,e,f,g,h,i,j,k,l;for(b=0;b<a.Q.f;b++){if(a.cb[b]==1||a.cb[b]==2){i=false;if(Jrb.Bo(a.Q,b)){for(h=0;h<Jrb.Gn(a.Q,b);h++){e=Jrb.Fn(a.Q,b,h);l=0;k=tX(Drb.NY,Qsb,6,3,15,1);for(j=0;j<Jrb.Gn(a.Q,e);j++){k[l]=Jrb.Fn(a.Q,e,j);k[l]!=b&&++l;}l==2&&a.e[k[0]]>a.e[k[1]]^k[0]<k[1]&&(i=!i);}}else {for(h=1;h<Jrb.Gn(a.Q,b);h++){for(j=0;j<h;j++){f=Jrb.Fn(a.Q,b,h);g=Jrb.Fn(a.Q,b,j);a.e[f]>a.e[g]&&(i=!i);f<g&&(i=!i);}}}Jrb.bm(a.Q,b,a.cb[b]==1^i?1:2,a.eb[b]);}else {Jrb.bm(a.Q,b,a.cb[b],a.eb[b]);}}for(c=0;c<a.Q.g;c++){if(a.n[c]==1||a.n[c]==2){i=false;for(h=0;h<2;h++){d=Jrb.Ik(a.Q,h,c);if(Jrb.Gn(a.Q,d)==3){k=tX(Drb.NY,Qsb,6,2,15,1);l=0;for(j=0;j<3;j++)Jrb.Fn(a.Q,d,j)!=Jrb.Ik(a.Q,1-h,c)&&(k[l++]=Jrb.Fn(a.Q,d,j));a.e[k[0]]>a.e[k[1]]&&(i=!i);k[0]<k[1]&&(i=!i);}}Jrb.qm(a.Q,c,a.n[c]==1^i?1:2,a.o[c]);}else {Jrb.qm(a.Q,c,a.n[c],a.o[c]);}}};Jrb.bg=function bg(a){var b;for(b=0;b<a.Q.f;b++){Jrb.fm(a.Q,b,a.L[b]);}};Jrb.cg=function cg(a){var b,c,d;for(b=0;b<a.Q.f;b++)!Jrb.fl(a.Q,b)&&a.cb[b]==3&&Jrb.Ul(a.Q,b,true);for(c=0;c<a.Q.g;c++){if(a.n[c]==3){d=Jrb.Qk(a.Q,c);d==1?Jrb.Ul(a.Q,Jrb.Ik(a.Q,0,c),true):d==2&&Jrb.tm(a.Q,c,386);}}};Jrb.dg=function dg(a,b){return a<b?a:a-b;};Jrb.eg=function eg(a){Jrb.fg.call(this,a,0);};Jrb.fg=function fg(a,b){this.Q=a;this.P=b;Jrb.ou(this.Q,7);this.c=Jrb.gg(this.Q.f);(this.P&Vtb)==0&&Jrb.Xe(this);this.ib=(b&64)!=0||Jrb.el(this.Q);this.b=Jrb.Vn(this.Q)==0;if((this.P&Vtb)==0){this.cb=tX(Drb.JY,nub,6,this.Q.f,15,1);this.eb=tX(Drb.c3,itb,6,this.Q.f,16,1);this.hb=tX(Drb.c3,itb,6,this.Q.f,16,1);this.n=tX(Drb.JY,nub,6,this.Q.g,15,1);this.p=tX(Drb.c3,itb,6,this.Q.g,16,1);this.o=tX(Drb.c3,itb,6,this.Q.g,16,1);}Jrb._e(this);(this.P&Vtb)==0&&Jrb.ff(this);Jrb.ef(this);};Jrb.gg=function gg(a){var b;b=0;while(a>0){a>>=1;++b;}return b;};Z3(29,1,{},Jrb.eg,Jrb.fg);_.b=false;_.c=0;_.q=false;_.s=0;_.u=0;_.v=0;_.F=false;_.H=0;_.J=false;_.K=false;_.N=0;_.P=0;_.S=0;_.T=0;_.Y=false;_.ib=false;Drb.dZ=B5(29);Jrb.hg=function hg(a,b){var c,d,e,f;if(a.d!=b.d)return a.d>b.d?1:-1;e=a.a.length;f=b.a.length;c=$wnd.Math.min(e,f);for(d=0;d<c;d++){--e;--f;if(a.a[e]!=b.a[f])return a.a[e]>b.a[f]?1:-1;}if(e!=f)return e>f?1:-1;if(a.b!=b.b)return a.b>b.b?1:-1;return 0;};Jrb.ig=function ig(a,b){return Jrb.hg(a,b);};Jrb.jg=function jg(){};Z3(257,1,{},Jrb.jg);_.qb=function kg(a,b){return Jrb.ig(a,b);};_.jb=function lg(a){return this===a;};Drb.WY=B5(257);Jrb.mg=function mg(){};Z3(152,1,{152:1},Jrb.mg);_.b=0;_.c=0;_.d=0;Drb.VY=B5(152);Jrb.ng=function ng(a,b){if(a.c!=b.c)return a.c>b.c?1:-1;return 0;};Jrb.og=function og(a,b){return Jrb.ng(a,b);};Jrb.pg=function pg(){};Z3(258,1,{},Jrb.pg);_.qb=function qg(a,b){return Jrb.og(a,b);};_.jb=function rg(a){return this===a;};Drb.YY=B5(258);Jrb.sg=function sg(){};Z3(153,1,{153:1},Jrb.sg);_.a=0;_.b=0;_.c=0;Drb.XY=B5(153);Jrb.tg=function tg(a,b,c){if(a.b==0){++a.c;a.b=63;}if(a.b==63){a.d[a.c]=G3(a.d[a.c],c);a.b-=b;}else {if(a.b>=b){a.d[a.c]=H3(a.d[a.c],b);a.d[a.c]=G3(a.d[a.c],c);a.b-=b;}else {a.d[a.c]=H3(a.d[a.c],a.b);a.d[a.c]=G3(a.d[a.c],I3(c,b-a.b));b-=a.b;++a.c;a.b=63-b;a.d[a.c]=G3(a.d[a.c],j3(c,Drb.r3((1<<b)-1)));}}};Jrb.ug=function ug(a,b){a.d[a.c]=i3(a.d[a.c],b);};Jrb.vg=function vg(a,b){var c;for(c=0;c<a.c;c++)if(E3(a.d[c],b.d[c]))return z3(a.d[c],b.d[c])?-1:1;return t3(a.d[a.c],b.d[a.c])?0:z3(a.d[a.c],b.d[a.c])?-1:1;};Jrb.wg=function wg(a,b){a.a=b;a.c=0;a.b=63;Krb.Heb(a.d);};Jrb.xg=function xg(a){this.d=tX(Drb.OY,iub,6,a,14,1);};Z3(118,1,{118:1,28:1},Jrb.xg);_.rb=function yg(a){return Jrb.vg(this,a);};_.a=0;_.b=0;_.c=0;Drb.ZY=B5(118);Jrb.zg=function zg(a,b){if(a.b!=b.b)return a.b>b.b?-1:1;if(a.c!=b.c)return a.c>b.c?-1:1;return 0;};Jrb.Ag=function Ag(a,b,c){this.b=$wnd.Math.max(a,b);this.c=$wnd.Math.min(a,b);this.a=c;};Z3(151,1,{151:1,28:1},Jrb.Ag);_.rb=function Bg(a){return Jrb.zg(this,a);};_.a=0;_.b=0;_.c=0;Drb.$Y=B5(151);Jrb.Cg=function Cg(a,b,c,d){this.a=(Hrb.cnb(b),Krb.Eeb(a,tX(Drb.NY,Qsb,6,b,15,1),0,b));this.b=(Hrb.cnb(d),Krb.Eeb(c,tX(Drb.NY,Qsb,6,d,15,1),0,d));};Z3(189,1,{189:1},Jrb.Cg);Drb._Y=B5(189);Jrb.Dg=function Dg(a,b){var c,d;for(d=0;d<a.g[b].length;d++){c=a.g[b][d];if(a.f[c]&&(a.o[c]==1||a.o[c]==2)&&a.k[c]==0)return true;}return false;};Jrb.Eg=function Eg(a,b,c){var d,e,f,g,h;e=0;g=0;for(h=0;h<a.g[b].length;h++){d=a.g[b][h];if(a.k[d]==c){f=1<<a.j[d];if((g&f)==0){g|=f;++e;}}}return e;};Jrb.Fg=function Fg(a){var b,c,d,e,f,g,h,i,j,k,l,m;k=new Krb.nkb();for(l=0;l<a.i.f;l++){if(Jrb.wn(a.i,l)<2||Jrb.Gn(a.i,l)>2){for(g=1;g<Jrb.Gn(a.i,l);g++){b=Jrb.Fn(a.i,l,g);for(h=0;h<g;h++){c=Jrb.Fn(a.i,l,h);Jrb.Kg(a,b,c)&&(i=Jrb.Rg(a,b,c),!!i&&Jrb.kq(i,a.f)&&Krb.pjb(k.a,i,(Erb.Q4(),P4))==null);}}}}for(m=0;m<a.i.g;m++){if(a.c[m]!=0){if(Jrb.Qk(a.i,m)!=2||a.c[m]!=2)continue;}b=Jrb.Ik(a.i,0,m);c=Jrb.Ik(a.i,1,m);Jrb.Kg(a,b,c)&&(i=Jrb.Rg(a,b,c),!!i&&Jrb.kq(i,a.f)&&Krb.pjb(k.a,i,(Erb.Q4(),P4))==null);}a.g=tX(Drb.NY,lub,7,k.a.c,0,2);a.e=tX(Drb.c3,itb,6,a.i.f,16,1);f=0;for(j=(e=new Krb.Hjb(new Krb.Njb(new Krb.ydb(k.a).a).b),new Krb.Hdb(e));Krb.gcb(j.a.a);){i=(d=Krb.Fjb(j.a),d.mc());a.g[f++]=i.b;for(g=0;g<i.b.length;g++)a.e[i.b[g]]=true;}};Jrb.Gg=function Gg(a,b,c,d){var e,f,g,h,i,j,k;e=tX(Drb.NY,Qsb,6,Jrb.Gn(a.i,c),15,1);h=0;for(g=0;g<Jrb.Gn(a.i,c);g++){e[h]=Jrb.Fn(a.i,c,g);!d[e[h]]&&Jrb.Kg(a,b,e[h])&&++h;}if(h==0)return  -1;if(h==1)return e[0];i=-1;j=xsb;for(f=0;f<h;f++){k=Jrb.go(a.i,b,e[f],xsb,d);if(k<j){j=k;i=e[f];}}return i;};Jrb.Hg=function Hg(a,b){return a.f[b]&&(a.o[b]==1||a.o[b]==2);};Jrb.Ig=function Ig(a,b){return a.e[b];};Jrb.Jg=function Jg(a){var b,c;c=true;for(b=0;b<a.i.f;b++){if(a.o[b]!=0&&!a.e[b]){c=false;break;}}return c;};Jrb.Kg=function Kg(a,b,c){var d,e,f,g,h;if(b==c)return false;if(a.a[b]!=a.a[c])return false;if(a.o[b]!=0){if(a.o[b]==3||a.o[c]==3)return false;if(a.p[b]^a.o[b]!==a.o[c])return false;if(a.k[b]!==a.k[c]||a.j[b]!==a.j[c])return false;}d=Jrb.Dn(a.i,b,c);if(d!=-1){if(Jrb.Qk(a.i,d)==1&&a.c[d]!=0)return false;if(Jrb.Qk(a.i,d)==2&&a.c[d]==1)return false;}if(Jrb.wn(a.i,b)==1&&!Jrb.yo(a.i,b)){e=-1;for(h=0;h<Jrb.Gn(a.i,b);h++){if(Jrb.Fn(a.i,b,h)!=c&&Jrb.In(a.i,b,h)==2){e=Jrb.Hn(a.i,b,h);break;}}f=-1;for(g=0;g<Jrb.Gn(a.i,c);g++){if(Jrb.Fn(a.i,c,g)!=b&&Jrb.In(a.i,c,g)==2){f=Jrb.Hn(a.i,c,g);break;}}if(e!=-1&&a.c[e]!=0&&a.d[e]^a.c[e]===a.c[f])return false;}return true;};Jrb.Lg=function Lg(a,b,c,d,e){var f,g,h,i,j,k;i=null;f=null;for(k=0;k<a.g[b].length;k++){g=a.g[b][k];a.f[g]&&(a.o[g]==1||a.o[g]==2)&&(a.k[g]==0?f=Jrb.Tg(f,(e[g]<<16)+g):a.k[g]==d&&a.j[g]==c&&(i=Jrb.Tg(i,(e[g]<<16)+g)));}h=Jrb.gh(i,f);if(h==0)return false;if(h<0){for(j=0;j<a.g[b].length;j++){g=a.g[b][j];if(a.f[g]&&(a.o[g]==1||a.o[g]==2)){if(a.k[g]==0){a.k[g]=d<<24>>24;a.j[g]=c<<24>>24;}else if(a.k[g]==d&&a.j[g]==c){a.k[g]=0;a.j[g]=-1;}}}}return true;};Jrb.Mg=function Mg(a,b){var c,d,e,f,g,h;if(!a.b)return false;e=false;for(f=a.b.a.length-1;f>=0;f--){d=false;g=Krb.Ji(a.b,f);g.a==2?d=Jrb.Lg(a,g.b,g.c,g.d,b):g.a==1&&(d=Jrb.Qg(a,g.b,b));if(d){Krb.Ri(a.b,g);for(h=0;h<a.g[g.b].length;h++){c=a.g[g.b][h];a.n[c]=false;}e=true;}}return e;};Jrb.Ng=function Ng(a){var b,c,d,e,f,g,h,i;if(a.g!=null){g=new Jrb.fh(a);a.b=new Krb.Yi();for(e=0;e<a.g.length;e++){d=Jrb.Yg(g,e);if(d==0){Jrb.Ug(g,e);h=Jrb.Eg(a,e,2);b=Jrb.Eg(a,e,1);c=Jrb.Dg(a,e);if(h==1&&b==1&&!c){Jrb.Pg(a,e,g.a+g.f++);Krb.Ei(a.b,new Jrb.uj(e,1,-1,-1));}if(h>0){if(c){Jrb.Og(a,e,g.i+g.g++,2);++h;}Krb.Ei(a.b,new Jrb.uj(e,1,-1,-1));}else if(b>0){c&&Jrb.Og(a,e,g.a+g.f++,1);Krb.Ei(a.b,new Jrb.uj(e,1,-1,-1));}else if(c){Jrb.Og(a,e,g.a+g.f++,1);Krb.Ei(a.b,new Jrb.uj(e,1,-1,-1));}}else if(d==1){if(Jrb.Dg(a,e)){f=Jrb.Xg(g,e);i=Jrb.Zg(g,e);Krb.Ei(a.b,new Jrb.uj(e,2,f,i));}else {Jrb.Ug(g,e);Krb.Ei(a.b,new Jrb.uj(e,1,-1,-1));}}}}};Jrb.Og=function Og(a,b,c,d){var e,f;for(f=0;f<a.g[b].length;f++){e=a.g[b][f];if(a.f[e]&&(a.o[e]==1||a.o[e]==2)&&a.k[e]==0){a.k[e]=d<<24>>24;a.j[e]=c<<24>>24;}}};Jrb.Pg=function Pg(a,b,c){var d,e;for(e=0;e<a.g[b].length;e++){d=a.g[b][e];if(a.k[d]==2){a.k[d]=1;a.j[d]=c<<24>>24;}}};Jrb.Qg=function Qg(a,b,c){var d,e,f,g,h,i,j,k;f=a.g[b];e=1;for(i=0;i<f.length;i++){d=f[i];if(a.f[d]&&a.k[d]==2){e=2;break;}}g=tX(Drb.NY,lub,7,32,0,2);for(j=0;j<f.length;j++){d=f[j];a.f[d]&&a.k[d]==e&&(g[a.j[d]]=Jrb.Tg(g[a.j[d]],(c[d]<<16)+d));}for(k=0;k<32;k++)g[k]!=null&&Hrb.Nmb(g[k],Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));Krb.Yeb(g,new Jrb.hh());if(Jrb.gh(g[0],g[1])==0)return false;for(h=0;h<g[0].length;h++){d=g[0][h]&Gsb;a.k[d]=0;a.j[d]=-1;}return true;};Jrb.Rg=function Rg(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v;i=tX(Drb.NY,Qsb,6,a.i.f,15,1);r=tX(Drb.NY,Qsb,6,a.i.f,15,1);o=tX(Drb.c3,itb,6,a.i.f,16,1);j=tX(Drb.c3,itb,6,a.i.f,16,1);s=new Jrb.lq(a.i.f);i[0]=b;r[b]=c;r[c]=-2;s.a[b]=true;s.a[c]=true;f=0;k=0;while(f<=k){g=i[f];if(r[g]==g){for(l=0;l<Jrb.Gn(a.i,g);l++){d=Jrb.Fn(a.i,g,l);if(!s.a[d]){if(Jrb.In(a.i,g,l)==2&&Jrb.Dk(a.i,d)<10){i[++k]=d;r[d]=d;j[d]=j[g]||Jrb.wn(a.i,d)==2;o[d]=j[g]&&!o[g];s.a[d]=true;}else if(j[g]&&o[g]){t=Jrb.Gg(a,d,r[g],s.a);if(t==-1)return null;i[++k]=d;r[d]=t;r[t]=-2;j[d]=false;s.a[d]=true;s.a[t]=true;}else if(Jrb.Mo(a.i,Jrb.Hn(a.i,g,l))){i[++k]=d;r[d]=d;j[d]=false;s.a[d]=true;if((Jrb.Dk(a.i,d)==6&&Jrb.wn(a.i,d)==0||Jrb.Dk(a.i,d)==7&&Jrb.mk(a.i,d)==1||Jrb.Dk(a.i,d)==14||Jrb.Dk(a.i,d)==15&&Jrb.Gn(a.i,d)>2||Jrb.Dk(a.i,d)==16&&Jrb.Gn(a.i,d)>2)&&Jrb.Gn(a.i,d)>2){h=false;for(p=1;p<Jrb.Gn(a.i,d);p++){u=Jrb.Fn(a.i,d,p);if(!s.a[u]){for(q=0;q<p;q++){v=Jrb.Fn(a.i,d,q);if(!s.a[v]){if(Jrb.Kg(a,u,v)){i[++k]=u;r[u]=v;r[v]=-2;j[u]=false;s.a[u]=true;s.a[v]=true;h=true;}}}}}if(!h)return null;}}}}}else {e=tX(Drb.c3,itb,6,Jrb.Gn(a.i,g),16,1);for(m=0;m<Jrb.Gn(a.i,g);m++){d=Jrb.Fn(a.i,g,m);if(s.a[d]){e[m]=r[d]==d;}else {for(p=0;p<Jrb.Gn(a.i,d);p++){if(Jrb.Fn(a.i,d,p)==r[g]){e[m]=true;break;}}}}for(n=0;n<Jrb.Gn(a.i,g);n++){if(e[n]){d=Jrb.Fn(a.i,g,n);if(s.a[d]){if(Jrb.Dn(a.i,d,r[g])==-1)return null;}else {i[++k]=d;r[d]=d;o[d]=false;j[d]=true;s.a[d]=true;}}}for(l=0;l<Jrb.Gn(a.i,g);l++){if(!e[l]){d=Jrb.Fn(a.i,g,l);if(!s.a[d]){t=Jrb.Gg(a,d,r[g],s.a);if(t==-1)return null;i[++k]=d;r[d]=t;r[t]=-2;j[d]=false;s.a[d]=true;s.a[t]=true;}}}}++f;}return s;};Jrb.Sg=function Sg(a,b,c,d,e,f,g,h,i,j){this.i=a;this.a=b;this.f=c;this.o=d;this.c=e;this.k=f;this.j=g;this.p=h;this.d=i;this.n=j;Jrb.Fg(this);};Jrb.Tg=function Tg(a,b){var c,d;d=tX(Drb.NY,Qsb,6,a==null?1:a.length+1,15,1);for(c=0;c<d.length-1;c++)d[c]=a[c];d[d.length-1]=b;return d;};Z3(307,1,{},Jrb.Sg);Drb.bZ=B5(307);Jrb.Ug=function Ug(a,b){var c,d,e,f,g,h;for(e=0;e<a.b;e++){if(a.e[e][b]&&a.c[e]!=-3){for(d=0;d<=a.j.g.length;d++){if(d!=b&&a.e[e][d]){a.e[e][b]=false;h=e<a.a?e:e<a.b?e-a.a:-1;g=Jrb.dh(a,e<a.a?1:e<a.b?2:0);for(f=0;f<a.j.g[b].length;f++){c=a.j.g[b][f];Jrb.Hg(a.j,c)&&a.j.j[c]==h&&(a.j.j[c]=g<<24>>24);}}}}}};Jrb.Vg=function Vg(a,b,c){var d,e,f,g,h;h=false;g=1;b[c]=g;d=true;while(d){d=false;for(e=0;e<a.b;e++){if(b[e]==g){for(f=0;f<a.b;f++){if(b[f]==0&&Jrb.bh(a,e,f)){if(a.c[f]==-2){b[f]=g+1;d=true;}else if(a.c[f]!=a.c[c]){b[f]=g+1;h=true;}}}}}++g;}return h;};Jrb.Wg=function Wg(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o;for(i=d+1;i<a.j.g.length;i++){if(i!=d&&a.e[b][i]&&a.e[c][i]){g=tX(Drb.NY,Qsb,6,2,15,1);g[0]=c;g[1]=b;return g;}}o=tX(Drb.NY,Qsb,6,a.b,15,1);k=tX(Drb.NY,Qsb,6,a.b,15,1);j=tX(Drb.NY,Qsb,6,a.b,15,1);f=0;l=0;j[0]=b;k[b]=1;while(f<=l){for(m=0;m<a.d[j[f]].length;m++){e=a.d[j[f]][m];if(e==c){if(f==0)continue;h=k[j[f]]+1;g=tX(Drb.NY,Qsb,6,h,15,1);g[0]=e;g[1]=j[f];for(n=2;n<h;n++)g[n]=o[g[n-1]];return g;}if(k[e]==0&&a.c[e]!=-3){k[e]=k[j[f]]+1;j[++l]=e;o[e]=j[f];}}++f;}return null;};Jrb.Xg=function Xg(a,b){var c;for(c=0;c<a.b;c++)if(a.e[c][b]&&a.c[c]==-3)return c<a.a?c:c<a.b?c-a.a:-1;return  -1;};Jrb.Yg=function Yg(a,b){var c,d;c=0;for(d=0;d<a.b;d++)a.e[d][b]&&a.c[d]==-3&&++c;return c;};Jrb.Zg=function Zg(a,b){var c;for(c=0;c<a.b;c++)if(a.e[c][b]&&a.c[c]==-3)return c<a.a?1:c<a.b?2:0;return  -1;};Jrb.$g=function $g(a,b){return b<a.a?b:b<a.b?b-a.a:-1;};Jrb._g=function _g(a,b){return b<a.a?1:b<a.b?2:0;};Jrb.ah=function ah(a,b){var c,d;d=a.j.k[b];c=a.j.j[b];return d==0?a.b:d==1?c:a.a+c;};Jrb.bh=function bh(a,b,c){var d;for(d=0;d<a.j.g.length;d++)if(a.e[b][d]&&a.e[c][d])return true;return false;};Jrb.dh=function dh(a,b){return b==1?a.a+a.f++:a.i+a.g++;};Jrb.eh=function eh(a,b){var c,d,e,f,g,h,i,j,k,l;k=xsb;i=-1;l=-1;j=-1;for(d=0;d<a.j.i.f;d++){if(Jrb.Hg(a.j,d)&&a.j.k[d]!=0){for(h=0;h<b.length;h++){e=Jrb.$g(a,b[h]);f=Jrb._g(a,b[h]);if(a.j.k[d]==f&&a.j.j[d]==e){if(k>a.j.a[d]+(f==1?Fsb:0)){k=a.j.a[d]+(f==1?Fsb:0);i=e;l=f;j=b[h];}}}}}for(c=0;c<a.j.i.f;c++){if(Jrb.Hg(a.j,c)&&a.j.k[c]==l&&a.j.j[c]==i){a.j.k[c]=0;a.j.j[c]=-1;}}for(g=0;g<a.j.g.length;g++)a.e[j][g]=false;};Jrb.fh=function fh(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q;this.j=a;for(d=0;d<a.i.f;d++){a.f[d]&&(a.o[d]==1||a.o[d]==2)&&(a.k[d]==1?this.a<=a.j[d]&&(this.a=1+a.j[d]):a.k[d]==2&&this.i<=a.j[d]&&(this.i=1+a.j[d]));}this.b=this.a+this.i;this.e=rX(Drb.c3,[ssb,itb],[11,6],16,[this.b+1,a.g.length+1],2);for(e=0;e<a.i.f;e++)a.f[e]&&(a.o[e]==1||a.o[e]==2)&&!a.e[e]&&(this.e[Jrb.ah(this,e)][a.g.length]=true);for(i=0;i<a.g.length;i++){for(q=0;q<a.g[i].length;q++){c=a.g[i][q];a.f[c]&&(a.o[c]==1||a.o[c]==2)&&(this.e[Jrb.ah(this,c)][i]=true);}}this.d=tX(Drb.NY,lub,7,this.b,0,2);for(j=0;j<a.g.length;j++){for(n=1;n<this.b;n++){if(this.e[n][j]){for(o=0;o<n;o++){if(this.e[o][j]){this.d[n]=Jrb.Tg(this.d[n],o);this.d[o]=Jrb.Tg(this.d[o],n);}}}}}this.c=tX(Drb.NY,Qsb,6,this.b+1,15,1);for(m=0;m<this.b;m++){this.e[m][a.g.length]?this.c[m]=-1:this.c[m]=-2;}for(k=0;k<a.g.length;k++){if(this.e[this.b][k]){for(l=0;l<this.b;l++){this.e[l][k]&&this.c[l]!=k&&(this.c[l]==-2?this.c[l]=k:this.c[l]=-3);}}}for(b=0;b<this.b;b++){if(this.c[b]>=-1){f=tX(Drb.NY,Qsb,6,this.b,15,1);if(Jrb.Vg(this,f,b)){for(l=0;l<this.b;l++){f[l]!=0&&(this.c[l]=-3);}}}}for(h=0;h<a.g.length-1;h++){for(n=1;n<this.b;n++){if(this.e[n][h]&&this.c[n]!=-3){for(o=0;o<n;o++){if(this.e[o][h]&&this.c[o]!=-3){g=Jrb.Wg(this,n,o,h);if(g!=null){for(p=0;p<g.length;p++)this.c[g[p]]=-3;Jrb.eh(this,g);break;}}}}}}};Z3(308,1,{},Jrb.fh);_.a=0;_.b=0;_.f=0;_.g=0;_.i=0;Drb.aZ=B5(308);Jrb.gh=function gh(a,b){var c,d;if(a==null)return b==null?0:1;if(b==null)return  -1;c=$wnd.Math.min(a.length,b.length);for(d=0;d<c;d++)if((a[d]&vub)!=(b[d]&vub))return (a[d]&vub)<(b[d]&vub)?-1:1;return a.length==b.length?0:a.length<b.length?-1:1;};Jrb.hh=function hh(){};Z3(309,1,{},Jrb.hh);_.qb=function ih(a,b){return Jrb.gh(a,b);};_.jb=function jh(a){return this===a;};Drb.cZ=B5(309);Jrb.kh=function kh(a,b){switch(b){case 0:return Jrb.lh(a,false);case 1:return Jrb.nh(a);case 2:return Jrb.mh(a);case 3:return Jrb.ph(a);case 4:return Jrb.oh(a);}return null;};Jrb.lh=function lh(b,c){try{if(!c)return Jrb.Kf(new Jrb.eg(b));b=Jrb.su(b);Jrb.dp(b,true);Jrb.xq(b);return Jrb.Kf(new Jrb.eg(b));}catch(a){a=g3(a);if(zY(a,23)){Erb.X7();return null;}else throw h3(a);}};Jrb.mh=function mh(b){var c,d;try{b=Jrb.su(b);Jrb.Cu(b);c=b.r;for(d=0;d<c;d++){b.H[d]=1;b.T=0;}return Jrb.Kf(new Jrb.eg(b));}catch(a){a=g3(a);if(zY(a,23)){Erb.X7();return null;}else throw h3(a);}};Jrb.nh=function nh(b){try{b=Jrb.su(b);Jrb.Cu(b);return Jrb.Kf(new Jrb.eg(b));}catch(a){a=g3(a);if(zY(a,23)){Erb.X7();return null;}else throw h3(a);}};Jrb.oh=function oh(b){var c;try{b=Jrb.su(b);Jrb.Cu(b);c=Jrb.Uu(new Jrb.Xu(b));return Jrb.Kf(new Jrb.fg(c,8));}catch(a){a=g3(a);if(zY(a,23)){Erb.X7();return null;}else throw h3(a);}};Jrb.ph=function ph(b){var c;try{c=Jrb.Uu(new Jrb.Xu(b));return Jrb.Kf(new Jrb.fg(c,8));}catch(a){a=g3(a);if(zY(a,23)){Erb.X7();return null;}else throw h3(a);}};Jrb.qh=function qh(a,b,c,d){a.a+=b;a.b+=c;a.c+=d;};Jrb.rh=function rh(a,b){a.a+=b.a;a.b+=b.b;a.c+=b.c;return a;};Jrb.sh=function sh(a,b){return new Jrb.Hh(a.a+b.a,a.b+b.b,a.c+b.c);};Jrb.th=function th(a,b){a.a=(a.a+b.a)/2;a.b=(a.b+b.b)/2;a.c=(a.c+b.c)/2;return a;};Jrb.uh=function uh(a,b){if(a.a!=b.a)return a.a<b.a?-1:1;if(a.b!=b.b)return a.b<b.b?-1:1;if(a.c!=b.c)return a.c<b.c?-1:1;return 0;};Jrb.vh=function vh(a,b){return new Jrb.Hh(a.b*b.c-a.c*b.b,-(a.a*b.c-a.c*b.a),a.a*b.b-a.b*b.a);};Jrb.wh=function wh(a,b){return $wnd.Math.sqrt((b.a-a.a)*(b.a-a.a)+(b.b-a.b)*(b.b-a.b)+(b.c-a.c)*(b.c-a.c));};Jrb.xh=function xh(a,b){var c;if(b==null||!zY(b,24))return false;c=b;return $wnd.Math.abs(c.a-a.a)+$wnd.Math.abs(c.b-a.b)+$wnd.Math.abs(c.c-a.c)<1.0E-6;};Jrb.yh=function yh(a,b){var c,d,e;d=a.a*a.a+a.b*a.b+a.c*a.c;e=b.a*b.a+b.b*b.b+b.c*b.c;if(d==0||e==0)return 0;c=(a.a*b.a+a.b*b.b+a.c*b.c)/$wnd.Math.sqrt(d*e);if(c>=1)return 0;if(c<=-1)return ntb;return $wnd.Math.acos(c);};Jrb.zh=function zh(a,b){var c,d,e;c=a.a;d=a.b;e=a.c;a.a=c*b[0][0]+d*b[1][0]+e*b[2][0];a.b=c*b[0][1]+d*b[1][1]+e*b[2][1];a.c=c*b[0][2]+d*b[1][2]+e*b[2][2];return a;};Jrb.Ah=function Ah(a,b){a.a*=b;a.b*=b;a.c*=b;return a;};Jrb.Bh=function Bh(a,b,c,d){a.a=b;a.b=c;a.c=d;};Jrb.Ch=function Ch(a,b){Jrb.Bh(a,b.a,b.b,b.c);return a;};Jrb.Dh=function Dh(a,b){a.a-=b.a;a.b-=b.b;a.c-=b.c;return a;};Jrb.Eh=function Eh(a,b){return new Jrb.Hh(a.a-b.a,a.b-b.b,a.c-b.c);};Jrb.Fh=function Fh(a){var b;b=$wnd.Math.sqrt(a.a*a.a+a.b*a.b+a.c*a.c);if(b==0){Erb.X7();a.a=1;a.b=0;a.c=0;return a;}a.a/=b;a.b/=b;a.c/=b;return a;};Jrb.Gh=function Gh(){};Jrb.Hh=function Hh(a,b,c){this.a=a;this.b=b;this.c=c;};Jrb.Ih=function Ih(a){Jrb.Hh.call(this,a.a,a.b,a.c);};Z3(24,1,{24:1,4:1,28:1},Jrb.Gh,Jrb.Hh,Jrb.Ih);_.rb=function Jh(a){return Jrb.uh(this,a);};_.jb=function Kh(a){return Jrb.xh(this,a);};_.mb=function Lh(){var a;a=new isb.Nab('0.00');return '['+isb.Lab(a,this.a)+', '+isb.Lab(a,this.b)+', '+isb.Lab(a,this.c)+']';};_.a=0;_.b=0;_.c=0;Drb.eZ=B5(24);Jrb.Mh=function Mh(a,b){b.c*=a.c;b.a=b.a*a.c+a.a;b.b=b.b*a.c+a.b;};Jrb.Nh=function Nh(a,b){b.a=b.a*a.c+a.a;b.b=b.b*a.c+a.b;};Jrb.Oh=function Oh(a,b){b.c=b.c*a.c+a.a;b.d=b.d*a.c+a.b;b.b*=a.c;b.a*=a.c;};Jrb.Ph=function Ph(a){a.a=0;a.b=0;a.c=1;};Jrb.Qh=function Qh(a,b){return b*a.c+a.a;};Jrb.Rh=function Rh(a,b){return b*a.c+a.b;};Jrb.Sh=function Sh(){Jrb.Ph(this);};Jrb.Th=function Th(a,b,c){var d,e,f,g;Jrb.Ph(this);e=b.b/a.b;g=b.a/a.a;f=0;f==0&&(f=24);d=f/c;this.c=$wnd.Math.min(d,$wnd.Math.min(e,g));this.a=b.c+b.b/2-this.c*(a.c+a.b/2);this.b=b.d+b.a/2-this.c*(a.d+a.a/2);};Z3(154,1,{},Jrb.Sh,Jrb.Th);_.mb=function Uh(){return 'DepictorTransformation Offset: '+this.a+','+this.b+' Scaling: '+this.c;};_.a=0;_.b=0;_.c=0;Drb.fZ=B5(154);Jrb.Vh=function Vh(a){var b,c,d,e;if(!Erb.k7(Hrb.rnb(a).substr(0,21),fub)||(d=Hrb.rnb(gub).length,!Erb.k7(Hrb.rnb(a).substr(Hrb.rnb(a).length-d,d),gub)))return null;c=Erb.p7(a,A7(34),21);if(c==-1)return null;e=(Hrb.lnb(21,c,Hrb.rnb(a).length),Hrb.rnb(a).substr(21,c-21));b=Erb.x7(a,21+Hrb.rnb(e).length+1,Hrb.rnb(a).length-17);if(Erb.k7(e,'arrow'))return new Urb.JF(b);if(Erb.k7(e,'text'))return new Jrb.kv(b);return null;};Krb.Bfb=function Bfb(a){return new Krb.ajb(a);};Krb.Cfb=function Cfb(a){return new lsb.vmb(a.tb());};Krb.Wh=function Wh(a,b,c){var d,e;for(e=a.sb();e.jc();){d=e.kc();if(HY(b)===HY(d)||b!=null&&Erb.Lc(b,d)){c&&e.lc();return true;}}return false;};Krb.Xh=function Xh(a,b){var c,d;Hrb.gnb(b);for(d=b.sb();d.jc();){c=d.kc();if(!a.contains(c)){return false;}}return true;};Krb.Yh=function Yh(a,b){var c,d,e;Hrb.gnb(b);c=false;for(d=a.sb();d.jc();){e=d.kc();if(b.contains(e)){d.lc();c=true;}}return c;};Krb.Zh=function Zh(a){var b,c,d;d=new Krb.fjb(', ','[',']');for(c=a.sb();c.jc();){b=c.kc();Krb.cjb(d,HY(b)===HY(a)?'(this Collection)':b==null?zsb:e4(b));}return !d.a?d.c:Hrb.rnb(d.e).length==0?d.a.a:d.a.a+(''+d.e);};Z3(42,1,wub);_.tb=function hi(){return new Krb.ajb(this);};_.ub=function ii(){return new lsb.vmb(this.tb());};_.add=function $h(a){throw h3(new Erb._7('Add not supported on this collection'));};_.addAll=function _h(a){var b,c,d;Hrb.gnb(a);b=false;for(d=a.sb();d.jc();){c=d.kc();b=b|this.add(c);}return b;};_.clear=function ai(){var a;for(a=this.sb();a.jc();){a.kc();a.lc();}};_.contains=function bi(a){return Krb.Wh(this,a,false);};_.containsAll=function ci(a){return Krb.Xh(this,a);};_.isEmpty=function di(){return this.size()==0;};_.remove=function ei(a){return Krb.Wh(this,a,true);};_.removeAll=function fi(a){return Krb.Yh(this,a);};_.retainAll=function gi(a){var b,c,d;Hrb.gnb(a);b=false;for(c=this.sb();c.jc();){d=c.kc();if(!a.contains(d)){c.lc();b=true;}}return b;};_.toArray=function ji(){return this.vb(tX(Drb.v0,Bsb,1,this.size(),5,1));};_.vb=function ki(a){var b,c,d,e;e=this.size();a.length<e&&(a=Hrb.Omb(new Array(e),a));d=a;c=this.sb();for(b=0;b<e;++b){xX(d,b,c.kc());}a.length>e&&Drb.wX(a,e,null);return a;};_.mb=function li(){return Krb.Zh(this);};Drb.L0=B5(42);Krb.vib=function vib(a){return new Krb.ajb(a);};Krb.wib=function wib(a){var b;return lsb.tmb(a.ub(),(b=lsb.Dlb(new lsb.Klb(),lsb.rlb(new lsb.Ilb(),new lsb.Glb(),new lsb.imb(),BX(nX(Drb.Z1,1),xub,79,0,[(lsb.vlb(),lsb.tlb)]))),lsb.zlb(b,new lsb.Nlb())));};Krb.mi=function mi(a,b){var c,d;for(c=0,d=a.size();c<d;++c){if(Krb.Kib(b,a.getAtIndex(c))){return c;}}return  -1;};Krb.ni=function ni(a,b,c){return new Krb.rcb(a,b,c);};Z3(77,42,yub);_.tb=function Ai(){return new Krb.ajb(this);};_.addAtIndex=function oi(a,b){throw h3(new Erb._7('Add not supported on this list'));};_.add=function pi(a){this.addAtIndex(this.size(),a);return true;};_.addAllAtIndex=function qi(a,b){var c,d,e;Hrb.gnb(b);c=false;for(e=b.sb();e.jc();){d=e.kc();this.addAtIndex(a++,d);c=true;}return c;};_.clear=function ri(){this.wb(0,this.size());};_.jb=function si(a){var b,c,d,e,f;if(a===this){return true;}if(!zY(a,72)){return false;}f=a;if(this.size()!=f.size()){return false;}e=f.sb();for(c=this.sb();c.jc();){b=c.kc();d=e.kc();if(!(HY(b)===HY(d)||b!=null&&Erb.Lc(b,d))){return false;}}return true;};_.lb=function ti(){return Krb.Efb(this);};_.indexOf=function ui(a){return Krb.mi(this,a);};_.sb=function vi(){return new Krb.jcb(this);};_.lastIndexOf=function wi(a){var b;for(b=this.size()-1;b>-1;--b){if(Krb.Kib(a,this.getAtIndex(b))){return b;}}return  -1;};_.removeAtIndex=function xi(a){throw h3(new Erb._7('Remove not supported on this list'));};_.wb=function yi(a,b){var c,d;d=new Krb.pcb(this,a);for(c=a;c<b;++c){Hrb.enb(d.a<d.c.size());d.c.getAtIndex(d.b=d.a++);Krb.icb(d);}};_.setAtIndex=function zi(a,b){throw h3(new Erb._7('Set not supported on this list'));};_.subList=function Bi(a,b){return Krb.ni(this,a,b);};Drb.S0=B5(77);function Q5(a){if(a==null){return false;}return a.$implements__java_lang_Cloneable||Array.isArray(a);}Krb.Ci=function Ci(a){a.a=tX(Drb.v0,Bsb,1,0,5,1);};Krb.Di=function Di(a,b,c){Hrb.inb(b,a.a.length);Hrb.Hmb(a.a,b,c);};Krb.Ei=function Ei(a,b){Hrb.Kmb(a.a,b);return true;};Krb.Fi=function Fi(a,b,c){var d,e;Hrb.inb(b,a.a.length);d=c.toArray();e=d.length;if(e==0){return false;}Hrb.Imb(a.a,b,d);return true;};Krb.Gi=function Gi(a,b){var c,d;c=b.toArray();d=c.length;if(d==0){return false;}Hrb.Imb(a.a,a.a.length,c);return true;};Krb.Hi=function Hi(a){Hrb.Mmb(a.a,0);};Krb.Ii=function Ii(a,b){return Krb.Li(a,b,0)!=-1;};Krb.Ji=function Ji(a,b){Hrb.fnb(b,a.a.length);return a.a[b];};Krb.Ki=function Ki(a,b){return Krb.Li(a,b,0);};Krb.Li=function Li(a,b,c){for(;c<a.a.length;++c){if(Krb.Kib(b,a.a[c])){return c;}}return  -1;};Krb.Mi=function Mi(a){return a.a.length==0;};Krb.Ni=function Ni(a){return new Krb.ueb(a);};Krb.Oi=function Oi(a,b){return Krb.Pi(a,b,a.a.length-1);};Krb.Pi=function Pi(a,b,c){for(;c>=0;--c){if(Krb.Kib(b,a.a[c])){return c;}}return  -1;};Krb.Qi=function Qi(a,b){var c;c=(Hrb.fnb(b,a.a.length),a.a[b]);Hrb.Lmb(a.a,b,1);return c;};Krb.Ri=function Ri(a,b){var c;c=Krb.Li(a,b,0);if(c==-1){return false;}Krb.Qi(a,c);return true;};Krb.Si=function Si(a,b,c){var d;Hrb.jnb(b,c,a.a.length);d=c-b;Hrb.Lmb(a.a,b,d);};Krb.Ti=function Ti(a,b,c){var d;d=(Hrb.fnb(b,a.a.length),a.a[b]);a.a[b]=c;return d;};Krb.Ui=function Ui(a){return a.a.length;};Krb.Vi=function Vi(a,b){Krb.Xeb(a.a,a.a.length,b);};Krb.Wi=function Wi(a){return Hrb.Emb(a.a);};Krb.Xi=function Xi(a,b){var c,d;d=a.a.length;b.length<d&&(b=Hrb.Omb(new Array(d),b));for(c=0;c<d;++c){xX(b,c,a.a[c]);}b.length>d&&Drb.wX(b,d,null);return b;};Krb.Yi=function Yi(){Krb.Ci(this);};Krb.Zi=function Zi(a){Krb.Ci(this);Hrb.anb(a>=0,'Initial capacity must not be negative');};Z3(18,77,zub,Krb.Yi,Krb.Zi);_.addAtIndex=function $i(a,b){Krb.Di(this,a,b);};_.add=function _i(a){return Krb.Ei(this,a);};_.addAllAtIndex=function aj(a,b){return Krb.Fi(this,a,b);};_.addAll=function bj(a){return Krb.Gi(this,a);};_.clear=function cj(){Krb.Hi(this);};_.contains=function dj(a){return Krb.Ii(this,a);};_.getAtIndex=function ej(a){return Krb.Ji(this,a);};_.indexOf=function fj(a){return Krb.Ki(this,a);};_.isEmpty=function gj(){return Krb.Mi(this);};_.sb=function hj(){return Krb.Ni(this);};_.lastIndexOf=function ij(a){return Krb.Oi(this,a);};_.removeAtIndex=function jj(a){return Krb.Qi(this,a);};_.remove=function kj(a){return Krb.Ri(this,a);};_.wb=function lj(a,b){Krb.Si(this,a,b);};_.setAtIndex=function mj(a,b){return Krb.Ti(this,a,b);};_.size=function nj(){return Krb.Ui(this);};_.toArray=function oj(){return Krb.Wi(this);};_.vb=function pj(a){return Krb.Xi(this,a);};Drb.h1=B5(18);Jrb.qj=function qj(a){var b,c,d;d=new Erb.S7();for(c=new Krb.ueb(a);c.a<c.c.a.length;){b=Krb.teb(c);Erb.P7(Erb.P7(d,fub+b.pb()+'"'+b.ob()+gub),rsb);}return d.a;};Jrb.rj=function rj(b){var c,d;Krb.Yi.call(this);try{if(b)for(d=new Krb.ueb(b);d.a<d.c.a.length;){c=Krb.teb(d);Krb.Ei(this,c.nb());}}catch(a){a=g3(a);if(!zY(a,15))throw h3(a);}};Jrb.sj=function sj(a){var b,c,d;Krb.Yi.call(this);if(a==null||Hrb.rnb(a).length==0)return;b=0;c=Erb.o7(a,A7(10));while(c!=-1){d=Jrb.Vh((Hrb.lnb(b,c,Hrb.rnb(a).length),Hrb.rnb(a).substr(b,c-b)));!!d&&(Hrb.Kmb(this.a,d),true);b=c+1;c=Erb.p7(a,A7(10),b);}};Z3(181,18,zub,Jrb.rj,Jrb.sj);_.mb=function tj(){return Jrb.qj(this);};Drb.gZ=B5(181);Jrb.uj=function uj(a,b,c,d){this.b=a;this.a=b;this.c=c;this.d=d;};Z3(92,1,{92:1},Jrb.uj);_.a=0;_.b=0;_.c=0;_.d=0;Drb.hZ=B5(92);Jrb.vj=function vj(a){var b,c,d,e,f,g;if(a.j!=0)return a.j;if(a.i&&Jrb.Dk(a.e,a.a)!=15&&Jrb.Dk(a.e,a.a)!=16){for(g=0;g<Jrb.rn(a.e,a.a);g++){f=Jrb.Hn(a.e,a.a,g);if(Jrb.Al(a.e,f,a.a)){Jrb.Fn(a.e,a.a,g)==a.b?a.j=Jrb.Tk(a.e,f)==257?3:1:a.j=Jrb.Tk(a.e,f)==257?1:3;return a.j;}}}b=Jrb.Hk(a.e,a.a,a.g);d=Jrb.Hk(a.e,a.a,a.b);d<b&&(d+=mtb);if(Jrb.rn(a.e,a.a)==2){c=d-b;if(c>3.0915926535897933&&c<3.191592653589793){a.j=-1;return a.j;}a.j=c<ntb?4:2;return a.j;}else {e=Jrb.Hk(a.e,a.a,a.d);e<b&&(e+=mtb);a.j=e<d?2:4;return a.j;}};Jrb.wj=function wj(a,b,c,d){var e,f,g,h;this.e=a;this.g=c;this.a=d;g=-1;for(h=0;h<Jrb.rn(this.e,this.a);h++){e=Jrb.Fn(this.e,this.a,h);f=Jrb.Hn(this.e,this.a,h);if(e==this.g){Jrb.Tk(this.e,f)==386&&(this.j=-1);continue;}if(Jrb.Al(this.e,f,this.a)){this.i&&(a.u[d]|=Fsb);this.i=true;}if(g==b[e]){this.d=e;this.f=true;this.c=Jrb.Mo(this.e,f);}else if(g<b[e]){g=b[e];this.d=this.b;this.b=e;}else {this.d=e;}}};Z3(105,1,{},Jrb.wj);_.a=0;_.b=0;_.c=false;_.d=0;_.f=false;_.g=0;_.i=false;_.j=0;Drb.iZ=B5(105);Jrb.Ej=function Ej(){Jrb.Ej=_3;Jrb.yj=BX(nX(Drb.B0,1),Esb,2,6,['?','H','He','Li','Be','B','C','N','O','F','Ne','Na','Mg','Al','Si','P','S','Cl','Ar','K','Ca','Sc','Ti','V','Cr','Mn','Fe','Co','Ni','Cu','Zn','Ga','Ge','As','Se','Br','Kr','Rb','Sr','Y','Zr','Nb','Mo','Tc','Ru','Rh','Pd','Ag','Cd','In','Sn','Sb','Te','I','Xe','Cs','Ba','La','Ce','Pr','Nd','Pm','Sm','Eu','Gd','Tb','Dy','Ho','Er','Tm','Yb','Lu','Hf','Ta','W','Re','Os','Ir','Pt','Au','Hg','Tl','Pb','Bi','Po','At','Rn','Fr','Ra','Ac','Th','Pa','U','Np','Pu','Am','Cm','Bk','Cf','Es','Fm','Md','No','Lr','Rf','Db','Sg','Bh','Hs','Mt','Ds','Rg','Cn','Nh','Fl','Mc','Lv','Ts','Og',Aub,Aub,Aub,Aub,Aub,Aub,Aub,Aub,Aub,Aub,'R4','R5','R6','R7','R8','R9','R10','R11','R12','R13','R14','R15','R16','R1','R2','R3','A','A1','A2','A3',Aub,Aub,'D','T','X','R','H2','H+','Nnn','HYD','Pol',Aub,Aub,Aub,Aub,Aub,Aub,Aub,Aub,Aub,Aub,Aub,'Ala','Arg','Asn','Asp','Cys','Gln','Glu','Gly','His','Ile','Leu','Lys','Met','Phe','Pro','Ser','Thr','Trp','Tyr','Val']);Jrb.Cj=BX(nX(Drb.b3,1),hub,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]);Jrb.Bj=BX(nX(Drb.JY,1),nub,6,15,[6]);Jrb.xj=BX(nX(Drb.JY,1),nub,6,15,[2]);Jrb.zj=BX(nX(Drb.JY,2),ssb,9,0,[null,BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[0]),BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[4]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[0]),BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[4]),BX(nX(Drb.JY,1),nub,6,15,[3,5]),BX(nX(Drb.JY,1),nub,6,15,[2,4,6]),BX(nX(Drb.JY,1),nub,6,15,[1,3,5,7]),BX(nX(Drb.JY,1),nub,6,15,[0]),BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),null,null,null,null,null,null,null,null,null,null,BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[2,4]),BX(nX(Drb.JY,1),nub,6,15,[3,5]),BX(nX(Drb.JY,1),nub,6,15,[2,4,6]),BX(nX(Drb.JY,1),nub,6,15,[1,3,5,7]),BX(nX(Drb.JY,1),nub,6,15,[0,2]),BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),null,null,null,null,null,null,null,null,null,null,BX(nX(Drb.JY,1),nub,6,15,[1,2,3]),BX(nX(Drb.JY,1),nub,6,15,[2,4]),BX(nX(Drb.JY,1),nub,6,15,[3,5]),BX(nX(Drb.JY,1),nub,6,15,[2,4,6]),BX(nX(Drb.JY,1),nub,6,15,[1,3,5,7]),BX(nX(Drb.JY,1),nub,6,15,[0,2,4,6]),BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[2])]);Jrb.Aj=BX(nX(Drb.JY,2),ssb,9,0,[null,BX(nX(Drb.JY,1),nub,6,15,[1]),null,BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),null,null,BX(nX(Drb.JY,1),nub,6,15,[-3]),BX(nX(Drb.JY,1),nub,6,15,[-2]),BX(nX(Drb.JY,1),nub,6,15,[-1]),null,BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),null,BX(nX(Drb.JY,1),nub,6,15,[-3]),BX(nX(Drb.JY,1),nub,6,15,[-2]),BX(nX(Drb.JY,1),nub,6,15,[-1]),null,BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2,3,4]),BX(nX(Drb.JY,1),nub,6,15,[2,3,4,5]),BX(nX(Drb.JY,1),nub,6,15,[2,3,6]),BX(nX(Drb.JY,1),nub,6,15,[2,3,4,7]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[1,2]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2,4]),BX(nX(Drb.JY,1),nub,6,15,[-3,3,5]),BX(nX(Drb.JY,1),nub,6,15,[-2]),BX(nX(Drb.JY,1),nub,6,15,[-1]),null,BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[4]),BX(nX(Drb.JY,1),nub,6,15,[3,5]),BX(nX(Drb.JY,1),nub,6,15,[6]),BX(nX(Drb.JY,1),nub,6,15,[4,6,7]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2,4]),BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2,4]),BX(nX(Drb.JY,1),nub,6,15,[-3,3,5]),BX(nX(Drb.JY,1),nub,6,15,[-2,4,6]),BX(nX(Drb.JY,1),nub,6,15,[-1]),null,BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3,4]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[4]),BX(nX(Drb.JY,1),nub,6,15,[5]),BX(nX(Drb.JY,1),nub,6,15,[6]),BX(nX(Drb.JY,1),nub,6,15,[4,6,7]),BX(nX(Drb.JY,1),nub,6,15,[3,4]),BX(nX(Drb.JY,1),nub,6,15,[3,4]),BX(nX(Drb.JY,1),nub,6,15,[2,4]),BX(nX(Drb.JY,1),nub,6,15,[1,3]),BX(nX(Drb.JY,1),nub,6,15,[1,2]),BX(nX(Drb.JY,1),nub,6,15,[1,3]),BX(nX(Drb.JY,1),nub,6,15,[2,4]),BX(nX(Drb.JY,1),nub,6,15,[3,5]),BX(nX(Drb.JY,1),nub,6,15,[-2,2,4]),BX(nX(Drb.JY,1),nub,6,15,[-1,1]),null,BX(nX(Drb.JY,1),nub,6,15,[1]),BX(nX(Drb.JY,1),nub,6,15,[2]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[4]),BX(nX(Drb.JY,1),nub,6,15,[4,5]),BX(nX(Drb.JY,1),nub,6,15,[3,4,5,6]),BX(nX(Drb.JY,1),nub,6,15,[3,4,5,6]),BX(nX(Drb.JY,1),nub,6,15,[3,4,5,6]),BX(nX(Drb.JY,1),nub,6,15,[3,4,5,6]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3,4]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[3]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[2,3]),BX(nX(Drb.JY,1),nub,6,15,[3])]);};Jrb.Fj=function Fj(a,b){var c,d;c=-1;d=0;if((a.G[b]&16)!=0){c=16;++d;}if((a.G[b]&64)!=0){c=64;++d;}if((a.G[b]&32)!=0){c=32;++d;}if((a.G[b]&4)!=0){c=4;++d;}if((a.G[b]&2)!=0){c=2;++d;}if((a.G[b]&8)!=0){c=8;++d;}if((a.G[b]&1)!=0){c=1;++d;}c!=-1&&c!=(a.H[b]&127)&&(a.H[b]=c);d<2&&(a.G[b]&=-128);};Jrb.Gj=function Gj(a,b,c,d){var e;e=Jrb.Hj(a,6);Jrb.Bh(a.J[e],b,c,d);return e;};Jrb.Hj=function Hj(a,b){a.q>=a.M&&Jrb.ym(a,a.M*2);a.C[a.q]=0;Jrb.jm(a,a.q,b);a.s[a.q]=0;a.u[a.q]=0;a.B[a.q]=0;a.w[a.q]=0;Jrb.Bh(a.J[a.q],0,0,0);a.v!=null&&(a.v[a.q]=null);a.t!=null&&(a.t[a.q]=null);a.T=0;return a.q++;};Jrb.Ij=function Ij(a,b,c){return Jrb.Jj(a,b,c,Jrb.tl(a,b)||Jrb.tl(a,c)?16:1);};Jrb.Jj=function Jj(a,b,c,d){var e;if(b==c)return  -1;for(e=0;e<a.r;e++){if(a.D[0][e]==b&&a.D[1][e]==c||a.D[0][e]==c&&a.D[1][e]==b){a.H[e]<d&&(a.H[e]=d);return e;}}a.r>=a.N&&Jrb.zm(a,a.N*2);a.D[0][a.r]=b;a.D[1][a.r]=c;a.H[a.r]=d;a.F[a.r]=0;a.G[a.r]=0;a.T=0;return a.r++;};Jrb.Kj=function Kj(a,b){return Jrb.Lj(a,b,b.q,b.r);};Jrb.Lj=function Lj(a,b,c,d){var e,f,g,h,i;a.K=a.K|b.K;f=tX(Drb.NY,Qsb,6,b.q,15,1);h=Jrb.Ml(a,1);i=Jrb.Ml(a,2);for(e=0;e<c;e++)f[e]=Jrb.Yj(b,a,e,h,i);for(g=0;g<d;g++)Jrb.Zj(b,a,g,h,i,f[b.D[0][g]],f[b.D[1][g]],false);a.L=a.L&&b.L;a.I=0;a.T=0;return f;};Jrb.Mj=function Mj(a,b,c,d,e,f,g,h){var i,j;i=Jrb.ik(a,b,c);if(i==-1){a.q>=a.M&&Jrb.ym(a,a.M*2);i=Jrb.Hj(a,d);Jrb.Bh(a.J[i],b,c,0);a.A[i]=e;Jrb.Ql(a,i,f);Jrb.dm(a,i,g);Jrb.Vl(a,i,h);return true;}j=Jrb.Tj(a,i,d,e,f,g);Jrb.Vl(a,i,h);return j;};Jrb.Nj=function Nj(a,b,c,d){var e;for(e=0;e<a.r;e++){if(a.D[0][e]==b&&a.D[1][e]==c||a.D[0][e]==c&&a.D[1][e]==b){Jrb.Vj(a,e,d);a.T=0;return e;}}a.r>=a.N&&Jrb.zm(a,a.N*2);a.D[0][a.r]=b;a.D[1][a.r]=c;a.H[a.r]=d;a.F[a.r]=0;a.G[a.r]=0;a.T=0;return a.r++;};Jrb.Oj=function Oj(a,b,c,d,e,f){var g,h,i;while(a.q+d>a.M)Jrb.ym(a,a.M*2);while(a.r+d>a.N)Jrb.zm(a,a.N*2);g=Jrb.ik(a,b,c);if(g!=-1)return Jrb.Pj(a,g,d,e,f);h=Jrb.jk(a,b,c);if(h!=-1)return Jrb.Qj(a,h,d,e,f);g=Jrb.Gj(a,b,c,0);i=ntb*(d-2)/d;Jrb.El(a,g,d,g,e,0,ntb-i,f);a.T=0;return true;};Jrb.Pj=function Pj(a,b,c,d,e){var f,g,h,i,j,k;if(d&&Jrb.ao(a,b)>1||!d&&Jrb.ao(a,b)>2)return false;g=0;f=tX(Drb.LY,Osb,6,4,15,1);for(i=0;i<a.r;i++){for(j=0;j<2;j++){if(a.D[j][i]==b){if(g==2){g=3;break;}f[g++]=Jrb.Hk(a,b,a.D[1-j][i]);}}if(g==3)break;}if(g==3)return false;k=g==1?f[0]+ntb:$wnd.Math.abs(f[0]-f[1])>ntb?(f[0]+f[1])/2:(f[0]+f[1])/2+ntb;h=ntb*(c-2)/c;Jrb.El(a,b,c,b,d,k-h/2,ntb-h,e);a.T=0;return true;};Jrb.Qj=function Qj(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p;j=tX(Drb.NY,Qsb,6,2,15,1);i=tX(Drb.LY,Osb,6,2,15,1);j[0]=a.D[0][b];j[1]=a.D[1][b];if(Jrb.ao(a,j[0])>3)return false;if(Jrb.ao(a,j[1])>3)return false;g=0;f=tX(Drb.LY,Osb,6,4,15,1);for(m=0;m<a.r;m++){if(m==b)continue;for(n=0;n<2;n++){for(o=0;o<2;o++){if(a.D[n][m]==j[o]){if(g==4){g=5;break;}f[g++]=Jrb.Hk(a,j[o],a.D[1-n][m]);}}if(g==5)break;}if(g==5)break;}if(g==5)return false;i[0]=Jrb.Hk(a,j[0],j[1]);if(i[0]<0){i[1]=i[0]+ntb;h=0;}else {i[1]=i[0];i[0]=i[1]-ntb;h=1;}p=0;for(l=0;l<g;l++){f[l]>i[0]&&f[l]<i[1]?--p:++p;}h=p>0?1-h:h;k=ntb*(c-2)/c;Jrb.El(a,j[h],c-1,j[1-h],d,i[p>0?0:1]+ntb-k,ntb-k,e);a.T=0;return true;};Jrb.Rj=function Rj(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;e=tX(Drb.NY,Qsb,6,b.q,15,1);i=Jrb.Ml(a,1);j=Jrb.Ml(a,2);for(d=0;d<b.q;d++){if(b.C[d]!=0||(b.t==null?null:b.t[d]==null?null:Erb.i7((n=b.t[d],hsb.Kab(),n)))!=null){e[d]=Jrb.Yj(b,a,d,i,j);}else {e[d]=c;if((b.T&8)!=0){for(h=0;h<b.r;h++){for(k=0;k<2;k++){if(b.D[k][h]==d){o=b.D[1-k][h];p=b.u[o]&3;l=false;if(p==1||p==2){for(f=0;f<b.r;f++){if(f!=h){for(m=0;m<2;m++){b.D[m][f]==o&&b.D[1-m][f]<d&&(l=!l);}}}}l&&Jrb.bm(b,o,p==1?2:1,(b.u[o]&4)!=0);}}}}}}for(g=0;g<b.r;g++){Jrb.Zj(b,a,g,i,j,e[b.D[0][g]],e[b.D[1][g]],false);}a.L=a.L&&b.L;a.I=0;a.T=0;return e;};Jrb.Sj=function Sj(a,b){var c,d,e,f,g,h,i,j,k;c=a.J[b[0]];d=a.J[b[1]];e=a.J[b[2]];f=a.J[b[3]];i=new Jrb.Hh(d.a-c.a,d.b-c.b,d.c-c.c);j=new Jrb.Hh(e.a-d.a,e.b-d.b,e.c-d.c);k=new Jrb.Hh(f.a-e.a,f.b-e.b,f.c-e.c);g=new Jrb.Hh(i.b*j.c-i.c*j.b,-(i.a*j.c-i.c*j.a),i.a*j.b-i.b*j.a);h=new Jrb.Hh(j.b*k.c-j.c*k.b,-(j.a*k.c-j.c*k.a),j.a*k.b-j.b*k.a);return -$wnd.Math.atan2($wnd.Math.sqrt(j.a*j.a+j.b*j.b+j.c*j.c)*(i.a*h.a+i.b*h.b+i.c*h.c),g.a*h.a+g.b*h.b+g.c*h.c);};Jrb.Tj=function Tj(a,b,c,d,e,f){if((c==1||c==151||c==152)&&Jrb.ao(a,b)>1)return false;a.B[b]=j3(a.B[b],-2);a.v!=null&&(a.v[b]=null);a.t!=null&&(a.t[b]=null);if(c==a.C[b]&&d==a.A[b]&&e==((a.u[b]&Bub)>>>27)-1&&f==(a.u[b]&48))return false;if(c==151||c==152){d=c-149;c=1;}a.u[b]&=960;a.C[b]=c;a.A[b]=d;a.s[b]=0;a.B[b]=0;Jrb.Ql(a,b,e);Jrb.dm(a,b,f);Jrb.Kl(a,a.w[b]);a.T=0;return true;};Jrb.Uj=function Uj(a,b,c){if(c){if(a.s[b]>8)return false;++a.s[b];}else {if(a.s[b]<-8)return false;--a.s[b];}a.T=0;return true;};Jrb.Vj=function Vj(a,b,c){var d,e,f,g,h;f=false;g=a.H[b];if(c==511){f=Jrb.bl(a,b);}else if(Jrb.ip(a,b,c)){if(c==257||c==129){d=Jrb.Fl(a,b,a.D[0][b]);e=Jrb.Fl(a,b,a.D[1][b]);if(c==g){if(d==e||e){h=a.D[0][b];a.D[0][b]=a.D[1][b];a.D[1][b]=h;f=true;}}else {if(!d&&e){h=a.D[0][b];a.D[0][b]=a.D[1][b];a.D[1][b]=h;}a.H[b]=c;f=true;}}else {a.H[b]=c;f=true;}}if(f){a.T=(g&127)==(c&127)?a.T&7:0;a.G[b]=0;}return f;};Jrb.Wj=function Wj(a){a.q=0;a.r=0;a.K=false;a.L=false;a.I=0;a.v=null;a.t=null;a.P=null;a.T=0;};Jrb.Xj=function Xj(a){var b,c,d,e,f,g,h,i;for(g=0;g<a.r;g++){if(a.H[g]==512){c=a.D[0][g];d=a.D[1][g];if(a.C[c]==-1^a.C[d]==-1){if(a.s[c]!=0&&a.s[d]!=0){if(a.s[c]<0^a.s[d]<0){if(a.s[c]<0){++a.s[c];--a.s[d];}else {--a.s[c];++a.s[d];}}}}}}i=tX(Drb.NY,Qsb,6,a.q,15,1);e=0;for(b=0;b<a.q;b++){if(a.C[b]==-1){i[b]=-1;continue;}if(e<b){a.C[e]=a.C[b];a.s[e]=a.s[b];a.A[e]=a.A[b];a.u[e]=a.u[b];a.B[e]=a.B[b];a.w[e]=a.w[b];Jrb.Ch(a.J[e],a.J[b]);a.v!=null&&(a.v[e]=a.v[b]);a.t!=null&&(a.t[e]=a.t[b]);}i[b]=e;++e;}a.q=e;h=0;for(f=0;f<a.r;f++){if(a.H[f]==512)continue;a.H[h]=a.H[f];a.F[h]=a.F[f];a.G[h]=a.G[f];a.D[0][h]=i[a.D[0][f]];a.D[1][h]=i[a.D[1][f]];++h;}a.r=h;return i;};Jrb.Yj=function Yj(a,b,c,d,e){var f,g,h;f=b.q;f>=b.M&&Jrb.ym(b,b.M*2);h=(a.u[c]&_tb)>>18;g=-1;h==1?d==-1?g=Jrb.Ml(b,h):g=$wnd.Math.min(31,d+((a.u[c]&_tb)>>18!=1&&(a.u[c]&_tb)>>18!=2?-1:(a.u[c]&Cub)>>20)):h==2&&(e==-1?g=Jrb.Ml(b,h):g=$wnd.Math.min(31,e+((a.u[c]&_tb)>>18!=1&&(a.u[c]&_tb)>>18!=2?-1:(a.u[c]&Cub)>>20)));b.C[f]=a.C[c];b.s[f]=a.s[c];b.A[f]=a.A[c];b.u[f]=a.u[c];b.B[f]=b.K?a.B[c]:0;Jrb.Ch(b.J[f],a.J[c]);b.w[f]=a.w[c];b.v!=null&&(b.v[f]=null);if(a.v!=null&&a.v[c]!=null&&b.K){b.v==null&&(b.v=tX(Drb.NY,lub,7,b.C.length,0,2));b.v[f]=Krb.Ceb(a.v[c],a.v[c].length);}b.t!=null&&(b.t[f]=null);if(a.t!=null&&a.t[c]!=null){b.t==null&&(b.t=tX(Drb.JY,ssb,9,b.C.length,0,2));b.t[f]=Krb.Aeb(a.t[c],a.t[c].length);}if(g!=-1){b.u[f]&=-32505857;b.u[f]|=g<<20;}++b.q;b.T=0;return f;};Jrb.Zj=function Zj(a,b,c,d,e,f,g,h){var i,j,k,l;j=b.r;j>=b.N&&Jrb.zm(b,b.N*2);l=(a.F[c]&768)>>8;k=-1;l==1&&(d==-1?k=Jrb.Ml(b,l):k=$wnd.Math.min(32,d+((a.F[c]&768)>>8!=1&&(a.F[c]&768)>>8!=2?-1:(a.F[c]&Dub)>>10)));l==2&&(e==-1?k=Jrb.Ml(b,l):k=$wnd.Math.min(32,e+((a.F[c]&768)>>8!=1&&(a.F[c]&768)>>8!=2?-1:(a.F[c]&Dub)>>10)));b.D[0][j]=f;b.D[1][j]=g;i=h&&a.yb(c)?8:a.H[c];b.H[j]=i;b.F[j]=a.F[c];b.G[j]=b.K?a.G[c]:0;if(k!=-1){b.F[j]&=-31745;b.F[j]|=k<<10;}++b.r;b.T=0;return j;};Jrb.$j=function $j(a,b,c,d,e,f,g){return Jrb.Zj(a,b,c,d,e,f==null?a.D[0][c]:f[a.D[0][c]],f==null?a.D[1][c]:f[a.D[1][c]],g);};Jrb._j=function _j(a,b){var c,d;b.v=null;b.t=null;b.K=a.K;b.q=0;for(c=0;c<a.q;c++)Jrb.Yj(a,b,c,0,0);b.r=0;for(d=0;d<a.r;d++)Jrb.Zj(a,b,d,0,0,a.D[0][d],a.D[1][d],false);a.xb(b);};Jrb.ak=function ak(a,b){b.K=a.K;b.L=a.L;b.S=a.S;b.I=a.I;b.P=a.P;b.T=a.T&24;};Jrb.bk=function bk(a,b){var c,d;for(c=0;c<a.r;c++)for(d=0;d<2;d++)a.D[d][c]==b&&(a.H[c]=512);Jrb.Kl(a,a.w[b]);a.C[b]=-1;a.v!=null&&(a.v[b]=null);a.t!=null&&(a.t[b]=null);Jrb.Xj(a);a.T=0;};Jrb.ck=function ck(a,b,c){var d,e;d=Jrb.ik(a,b,c);if(d!=-1){(a.u[d]&512)!=0?Jrb.hk(a):Jrb.bk(a,d);a.T=0;return true;}e=Jrb.jk(a,b,c);if(e!=-1){(a.u[a.D[0][e]]&a.u[a.D[1][e]]&512)!=0?Jrb.hk(a):Jrb.fk(a,e);a.T=0;return true;}return false;};Jrb.dk=function dk(a,b){var c,d,e,f;if(b.length==0)return null;for(d=b,e=0,f=d.length;e<f;++e){c=d[e];a.C[c]=-1;}return Jrb.gk(a);};Jrb.ek=function ek(a,b){a.H[b]=512;Jrb.Xj(a);a.T=0;};Jrb.fk=function fk(a,b){var c,d,e;for(d=0;d<2;d++){c=0;for(e=0;e<a.r;e++){if(e==b)continue;(a.D[0][e]===a.D[d][b]||a.D[1][e]===a.D[d][b])&&++c;}if(c==0){Jrb.Kl(a,a.w[a.D[d][b]]);a.C[a.D[d][b]]=-1;}}a.H[b]=512;Jrb.Xj(a);a.T=0;};Jrb.gk=function gk(a){var b,c,d;d=false;for(b=0;b<a.q;b++){if(a.C[b]==-1){d=true;Jrb.Kl(a,a.w[b]);}}for(c=0;c<a.r;c++){if(a.H[c]==512){d=true;}else if(a.C[a.D[0][c]]==-1||a.C[a.D[1][c]]==-1){a.H[c]=512;d=true;}}if(d){a.T=0;return Jrb.Xj(a);}return null;};Jrb.hk=function hk(a){var b,c;c=false;for(b=0;b<a.q;b++){if((a.u[b]&512)!=0){a.C[b]=-1;c=true;}}return c&&Jrb.gk(a)!=null;};Jrb.ik=function ik(a,b,c){var d,e,f,g,h,i,j,k;g=-1;e=Jrb.Fk(a,a.q,a.r,Jrb.Dj);h=Eub;i=e*e/12;for(d=0;d<a.q;d++){j=a.J[d].a;k=a.J[d].b;f=(b-j)*(b-j)+(c-k)*(c-k);if(f<i&&f<h){h=f;g=d;}}return g;};Jrb.jk=function jk(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;m=-1;o=Jrb.Fk(a,a.q,a.r,Jrb.Dj);n=Eub;for(d=0;d<a.r;d++){p=a.J[a.D[0][d]].a;r=a.J[a.D[0][d]].b;q=a.J[a.D[1][d]].a;s=a.J[a.D[1][d]].b;k=q-p;l=s-r;e=$wnd.Math.sqrt(k*k+l*l);f=(p+q)/2;g=(r+s)/2;k=b-f;l=c-g;if($wnd.Math.sqrt(k*k+l*l)>e/2)continue;if(q==p)j=$wnd.Math.abs(p-b);else {h=(s-r)/(p-q);i=-h*p-r;j=$wnd.Math.abs((h*b+c+i)/$wnd.Math.sqrt(h*h+1));}if(j<o&&j<n){n=j;m=d;}}return m;};Jrb.kk=function kk(a,b){return ((a.u[b]&Bub)>>>27)-1;};Jrb.lk=function lk(a,b){return (a.u[b]&49152)>>14;};Jrb.mk=function mk(a,b){return a.s[b];};Jrb.nk=function nk(a,b){return a.u[b]&448;};Jrb.ok=function ok(a,b){var c;return a.t==null?null:a.t[b]==null?null:Erb.i7((c=a.t[b],hsb.Kab(),c));};Jrb.pk=function pk(a,b){return a.t==null?null:a.t[b];};Jrb.qk=function qk(a,b){return (a.u[b]&_tb)>>18!=1&&(a.u[b]&_tb)>>18!=2?-1:(a.u[b]&Cub)>>20;};Jrb.rk=function rk(a,b){return (a.u[b]&_tb)>>18;};Jrb.sk=function sk(a,b){return Jrb.yj[a.C[b]];};Jrb.tk=function tk(a,b){return a.v==null?null:a.v[b];};Jrb.uk=function uk(a,b){var c,d,e;if(a.v==null||a.v[b]==null)return E3(j3(a.B[b],1),0)?'':Jrb.yj[a.C[b]];e='';for(d=0;d<a.v[b].length;d++){d>0&&(e=(Hrb.gnb(e),e+(Hrb.gnb(','),',')));c=a.v[b][d];e=Erb.g7(e,Jrb.yj[c]);}return e;};Jrb.vk=function vk(a,b){return $wnd.Math.abs(a.w[b]);};Jrb.wk=function wk(a,b){return a.A[b];};Jrb.xk=function xk(a,b){return a.u[b]&3;};Jrb.yk=function yk(a,b){return a.B[b];};Jrb.zk=function zk(a,b){return a.u[b]&48;};Jrb.Ak=function Ak(a,b){return a.J[b].a;};Jrb.Bk=function Bk(a,b){return a.J[b].b;};Jrb.Ck=function Ck(a,b){return a.J[b].c;};Jrb.Dk=function Dk(a,b){return a.C[b];};Jrb.Ek=function Ek(a){return Jrb.Fk(a,a.q,a.r,Jrb.Dj);};Jrb.Fk=function Fk(a,b,c,d){return Jrb.Gk(a,b,c,d,a.J);};Jrb.Gk=function Gk(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o;l=false;m=0;for(j=0;j<c;j++)a.H[j]!=16&&(a.G[j]&Xtb)==0&&++m;if(m==0){for(k=0;k<c;k++)(a.G[k]&Xtb)==0&&++m;l=true;}if(m==0){if(b<2)return d;o=Eub;for(f=1;f<b;f++){for(g=0;g<f;g++){n=Jrb.wh(e[f],e[g]);n>0&&n<o&&(o=n);}}return o!=Eub?0.6*o:d;}h=0;for(i=0;i<c;i++){(l||a.H[i]!=16)&&(a.G[i]&Xtb)==0&&(h+=Jrb.wh(e[a.D[1][i]],e[a.D[0][i]]));}return h/m;};Jrb.Hk=function Hk(a,b,c){return Jrb.Vm(a.J[b].a,a.J[b].b,a.J[c].a,a.J[c].b);};Jrb.Ik=function Ik(a,b,c){return a.D[b][c];};Jrb.Jk=function Jk(a,b){return ((a.G[b]&7680)>>9)+((a.G[b]&122880)>>13);};Jrb.Kk=function Kk(a,b){return (a.G[b]&7680)>>9;};Jrb.Lk=function Lk(a,b){return (a.F[b]&48)>>4;};Jrb.Mk=function Mk(a,b){return (a.F[b]&768)>>8!=1&&(a.F[b]&768)>>8!=2?-1:(a.F[b]&Dub)>>10;};Jrb.Nk=function Nk(a,b){return (a.F[b]&768)>>8;};Jrb.Ok=function Ok(a,b){var c,d,e,f;c=a.D[0][b];d=a.D[1][b];e=a.J[d].a-a.J[c].a;f=a.J[d].b-a.J[c].b;return $wnd.Math.sqrt(e*e+f*f);};Jrb.Pk=function Pk(a,b,c){var d;for(d=0;d<a.r;d++)if(a.D[0][d]==b&&a.D[1][d]==c||a.D[0][d]==c&&a.D[1][d]==b)if(a.H[d]!=512)return d;return  -1;};Jrb.Qk=function Qk(a,b){if(a.K&&(a.G[b]&31)!=0){if((a.G[b]&9)!=0)return 1;if((a.G[b]&2)!=0)return 2;if((a.G[b]&4)!=0)return 3;if((a.G[b]&32)!=0)return 4;if((a.G[b]&64)!=0)return 5;if((a.G[b]&16)!=0)return 0;}switch(a.H[b]&127){case 1:case 8:return 1;case 2:return 2;case 4:return 3;case 32:return 4;case 64:return 5;default:return 0;}};Jrb.Rk=function Rk(a,b){return a.F[b]&3;};Jrb.Sk=function Sk(a,b){return a.G[b];};Jrb.Tk=function Tk(a,b){return a.H[b];};Jrb.Uk=function Uk(a,b){return a.H[b]&127;};Jrb.Vk=function Vk(a,b){var c,d,e,f,g;if(a.q==0)return null;d=a.J[0].a;f=a.J[0].b;e=a.J[0].a;g=a.J[0].b;for(c=1;c<a.q;c++){d>a.J[c].a?d=a.J[c].a:e<a.J[c].a&&(e=a.J[c].a);f>a.J[c].b?f=a.J[c].b:g<a.J[c].b&&(g=a.J[c].b);}if(!b){b=new Wrb.LG(d,f,e-d,g-f);}else {b.c=d;b.d=f;b.b=e-d;b.a=g-f;}return b;};Jrb.Wk=function Wk(a,b){var c;c=a.C[b]<Jrb.zj.length?Jrb.zj[a.C[b]]:null;return c==null?6:c[c.length-1];};Jrb.Xk=function Xk(a,b,c){return Jrb.Yk(a,b,c,a.C[b]);};Jrb.Yk=function Yk(a,b,c,d){var e,f;if(d>=171&&d<=190)return 0;f=0;(a.u[b]&48)==32&&(f-=1);((a.u[b]&48)==16||(a.u[b]&48)==48)&&(f-=2);e=a.s[b];if(e==0&&a.K){t3(j3(a.B[b],wtb),ztb)&&(e=-1);t3(j3(a.B[b],wtb),ytb)&&(e=1);}d==7||d==8||d==9?f+=e:d==6||d==14||d==32?f-=$wnd.Math.abs(e):d==15||d==33?c-f-e<=3?f+=e:f-=e:d==16||d==34||d==52?c-f-e<=4?f+=e:f-=$wnd.Math.abs(e):d==17||d==35||d==53?c-f-e<=5?f+=e:f-=$wnd.Math.abs(e):f-=e;return f;};Jrb.Zk=function Zk(a,b){var c;c=Jrb.$k(a,b);return c+Jrb.Yk(a,b,c,a.C[b]);};Jrb.$k=function $k(a,b){var c,d;c=((a.u[b]&Bub)>>>27)-1;c==-1&&(c=(d=a.C[b]<Jrb.zj.length?Jrb.zj[a.C[b]]:null,d==null?6:d[d.length-1]));return c;};Jrb._k=function _k(a,b){var c,d,e,f,g;f=Jrb.Bl(a,a.D[0][b])||Jrb.Bl(a,a.D[1][b])?5:3;for(d=0;d<2;d++){c=a.D[d][b];e=Jrb.Qk(a,b)+(g=Jrb.$k(a,c),g+Jrb.Yk(a,c,g,a.C[c]))-Jrb.ao(a,c);f>e&&(f=e);}return f;};Jrb.al=function al(a,b){return (a.u[b]&Fsb)!=0;};Jrb.bl=function bl(a,b){var c,d,e;d=Jrb._k(a,b);c=Jrb.tl(a,a.D[0][b])||Jrb.tl(a,a.D[1][b]);e=c?16:1;if(a.H[b]==64){a.H[b]=e;a.T=0;return true;}if(a.H[b]==32){a.H[b]=d>4?64:e;a.T=0;return true;}if(a.H[b]==4){a.H[b]=d>3?32:e;a.T=0;return true;}if(a.H[b]==2){a.H[b]=386;a.T&=7;if((a.F[b]&128)==0)return true;}if(a.H[b]==386){d>2?a.H[b]=4:a.H[b]=e;a.T=0;return true;}if((384&a.H[b])!=0){a.H[b]=1;a.T&=7;return true;}if(!c&&d<2)return false;if(a.H[b]==1){a.H[b]=2;a.T=0;return true;}if(d<1)return false;if(a.H[b]==16){a.H[b]=1;a.T=0;return true;}return false;};Jrb.cl=function cl(a){var b;a.T=0;a.C=tX(Drb.NY,Qsb,6,a.M,15,1);a.s=tX(Drb.NY,Qsb,6,a.M,15,1);a.w=tX(Drb.NY,Qsb,6,a.M,15,1);a.J=tX(Drb.eZ,Fub,24,a.M,0,1);for(b=0;b<a.M;b++)a.J[b]=new Jrb.Gh();a.A=tX(Drb.NY,Qsb,6,a.M,15,1);a.u=tX(Drb.NY,Qsb,6,a.M,15,1);a.B=tX(Drb.OY,iub,6,a.M,14,1);a.v=null;a.t=null;a.D=rX(Drb.NY,[lub,Qsb],[7,6],15,[2,a.N],2);a.H=tX(Drb.NY,Qsb,6,a.N,15,1);a.F=tX(Drb.NY,Qsb,6,a.N,15,1);a.G=tX(Drb.NY,Qsb,6,a.N,15,1);};Jrb.dl=function dl(a,b){a.T&=~b;};Jrb.el=function el(a){var b;for(b=0;b<a.q;b++)if(a.J[b].c!=0)return true;return false;};Jrb.fl=function fl(a,b){return (a.u[b]&Gub)!=0;};Jrb.gl=function gl(a,b){return a.C[b]==-1;};Jrb.hl=function hl(a,b){return (a.u[b]&4)!=0;};Jrb.il=function il(a,b){return (a.u[b]&Hub)!=0;};Jrb.jl=function jl(a,b){return a.w[b]<0;};Jrb.kl=function kl(a,b){return (a.F[b]&Iub)!=0;};Jrb.ll=function ll(a,b){return (a.G[b]&Xtb)!=0;};Jrb.ml=function ml(a,b){return (a.F[b]&Fsb)!=0;};Jrb.nl=function nl(a,b){return a.H[b]==512;};Jrb.ol=function ol(a,b){return (a.F[b]&4)!=0;};Jrb.pl=function pl(a,b){return (a.F[b]&oub)!=0;};Jrb.ql=function ql(a,b){var c,d,e,f;if(a.K){if(E3(j3(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!Jrb.Ym(c))return false;}}return Jrb.Ym(a.C[b]);};Jrb.rl=function rl(a,b){var c,d,e,f;if(a.K){if(E3(j3(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!Jrb.Zm(c))return false;}}return Jrb.Zm(a.C[b]);};Jrb.sl=function sl(a,b){return (a.u[b]&oub)!=0;};Jrb.tl=function tl(a,b){var c,d,e,f;if(a.K){if(E3(j3(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!(c>=3&&c<=4||c>=11&&c<=13||c>=19&&c<=31||c>=37&&c<=51||c>=55&&c<=84||c>=87&&c<=112))return false;}}return Jrb.$m(a.C[b]);};Jrb.ul=function ul(a,b){return a.A[b]==0;};Jrb.vl=function vl(a,b){var c,d,e,f;if(a.K){if(E3(j3(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!(c==1||c>=5&&c<=9||c>=14&&c<=17||c>=32&&c<=35||c>=52&&c<=53))return false;}}return Jrb._m(a.C[b]);};Jrb.wl=function wl(a){var b;for(b=0;b<a.q;b++){switch(a.C[b]){case 1:case 5:case 6:case 7:case 8:case 9:case 14:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:continue;default:return false;}}return true;};Jrb.xl=function xl(a,b){return (a.u[b]&512)!=0;};Jrb.yl=function yl(a,b){return (a.u[a.D[0][b]]&a.u[a.D[1][b]]&512)!=0;};Jrb.zl=function zl(a,b){return a.H[b]==257||a.H[b]==129;};Jrb.Al=function Al(a,b,c){return (a.H[b]==257||a.H[b]==129)&&a.D[0][b]==c;};Jrb.Bl=function Bl(a,b){var c,d,e,f;if(a.K){if(E3(j3(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!(c>=21&&c<=30||c>=39&&c<=48||c==57||c>=72&&c<=80||c==89||c>=104&&c<=112))return false;}}return Jrb.an(a.C[b]);};Jrb.Cl=function Cl(a,b){a.C[b]=-1;};Jrb.Dl=function Dl(a,b){a.H[b]=512;};Jrb.El=function El(a,b,c,d,e,f,g,h){var i,j,k,l,m,n,o,p,q,r;if(b!=d){q=a.J[b].a-a.J[d].a;r=a.J[b].b-a.J[d].b;h=$wnd.Math.sqrt(q*q+r*r);}i=b;k=Jrb.Dm(a,b)!=3;for(p=1;p<c;p++){m=a.J[i].a+h*$wnd.Math.sin(f);n=a.J[i].b+h*$wnd.Math.cos(f);o=-1;for(l=0;l<a.q;l++){if($wnd.Math.abs(m-a.J[l].a)<4&&$wnd.Math.abs(n-a.J[l].b)<4){o=l;break;}}if(o==-1){o=Jrb.Gj(a,m,n,0);a.J[o].a=m;a.J[o].b=n;a.J[o].c=0;}j=Jrb.Pk(a,i,o);if(j==-1){j=Jrb.Jj(a,i,o,Jrb.tl(a,i)||Jrb.tl(a,o)?16:1);if(e){k&&Jrb.Dm(a,a.D[0][j])<4&&Jrb.Dm(a,a.D[1][j])<3&&(a.H[j]=2);k=!k;}}i=o;f+=g;}j=Jrb.Pk(a,i,d);j==-1&&(j=Jrb.Jj(a,i,d,Jrb.tl(a,i)||Jrb.tl(a,d)?16:1));e&&k&&Jrb.Dm(a,a.D[0][j])<4&&Jrb.Dm(a,a.D[1][j])<4&&(a.H[j]=2);};Jrb.Fl=function Fl(a,b,c){var d,e;if(Jrb.Qk(a,b)!=1)return false;if((a.u[c]&3)!=0)return true;for(e=0;e<a.r;e++)if(e!=b&&a.H[e]==2&&(a.D[0][e]==c&&(a.u[a.D[1][e]]&3)!=0||a.D[1][e]==c&&(a.u[a.D[0][e]]&3)!=0))return true;for(d=0;d<a.r;d++)if(d!=b&&a.H[d]==1&&(a.D[0][d]==c||a.D[1][d]==c)&&(a.F[d]&3)!=0)return true;return false;};Jrb.Gl=function Gl(a){var b;for(b=0;b<a.q;b++)a.u[b]&=-449;};Jrb.Hl=function Hl(a){var b;for(b=0;b<a.q;b++)a.u[b]&=Jub;};Jrb.Il=function Il(a){var b;for(b=0;b<a.q;b++)a.u[b]&=-513;};Jrb.Jl=function Jl(a){var b;for(b=0;b<a.r;b++)a.F[b]&=-98305;};Jrb.Kl=function Kl(a,b){var c;for(c=0;c<a.q;c++)$wnd.Math.abs(a.w[c])==$wnd.Math.abs(b)&&(a.w[c]=0);};Jrb.Ll=function Ll(a){var b,c,d,e;e=false;for(c=0;c<a.q;c++){if(E3(j3(a.B[c],jtb),0)){a.C[c]=-1;e=true;}}e&&Jrb.gk(a);if(a.v!=null){a.v=null;e=true;}for(b=0;b<a.q;b++){if(E3(a.B[b],0)){a.B[b]=0;e=true;}}for(d=0;d<a.r;d++){if(a.G[d]!=0){a.G[d]=0;e=true;}if(a.H[d]==8){a.H[d]=1;e=true;}}e&&(a.T=0);return e;};Jrb.Ml=function Ml(a,b){var c,d,e,f,g,h,i,j,k;if(b==0)return 0;h=null;for(d=0;d<a.q;d++){if((a.u[d]&_tb)>>18==b){h==null&&(h=tX(Drb.c3,itb,6,32,16,1));h[(a.u[d]&_tb)>>18!=1&&(a.u[d]&_tb)>>18!=2?-1:(a.u[d]&Cub)>>20]=true;}}for(f=0;f<a.r;f++){if((a.F[f]&768)>>8==b){h==null&&(h=tX(Drb.c3,itb,6,32,16,1));h[(a.F[f]&768)>>8!=1&&(a.F[f]&768)>>8!=2?-1:(a.F[f]&Dub)>>10]=true;}}k=0;if(h!=null){j=tX(Drb.NY,Qsb,6,32,15,1);for(i=0;i<32;i++)h[i]&&(j[i]=k++);for(c=0;c<a.q;c++){if((a.u[c]&_tb)>>18==b){g=j[(a.u[c]&_tb)>>18!=1&&(a.u[c]&_tb)>>18!=2?-1:(a.u[c]&Cub)>>20];a.u[c]&=-32505857;a.u[c]|=g<<20;}}for(e=0;e<a.r;e++){if((a.F[e]&768)>>8==b){g=j[(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&Dub)>>10];a.F[e]&=-31745;a.F[e]|=g<<10;}}}return k;};Jrb.Nl=function Nl(a,b){var c;for(c=0;c<a.q;c++){a.J[c].a*=b;a.J[c].b*=b;}};Jrb.Ol=function Ol(a,b){a.q=b;a.T=0;};Jrb.Pl=function Pl(a,b){a.r=b;a.T=0;};Jrb.Ql=function Ql(a,b,c){if(c>=-1&&c<=14){a.u[b]&=-2013265921;a.u[b]|=1+c<<27;if(a.C[b]==6){if(c==-1||c==0||c==2||c==4){a.u[b]&=-49;c==2&&(a.u[b]|=16);}}}};Jrb.Rl=function Rl(a,b,c){a.u[b]&=-49153;a.u[b]|=c<<14;};Jrb.Sl=function Sl(a,b,c){a.s[b]=c;a.T=0;};Jrb.Tl=function Tl(a,b,c){a.u[b]&=-449;a.u[b]|=c;};Jrb.Ul=function Ul(a,b,c){c?a.u[b]|=Gub:a.u[b]&=Kub;a.T&=7;};Jrb.Vl=function Vl(a,b,c){var d,e;if(c!=null){if(Hrb.rnb(c).length==0)c=null;else {d=Jrb.Xm(c,321);if(d!=0&&d==a.C[b]||Erb.k7(c,'?')){Jrb.jm(a,b,d);c=null;}}}if(c==null){a.t!=null&&(a.t[b]=null);}else {a.t==null&&(a.t=tX(Drb.JY,ssb,9,a.M,0,2));a.t[b]=Hrb.Umb((e=c,hsb.Kab(),e));}};Jrb.Wl=function Wl(a,b,c){c!=null&&c.length==0&&(c=null);if(c==null){a.t!=null&&(a.t[b]=null);}else {a.t==null&&(a.t=tX(Drb.JY,ssb,9,a.M,0,2));a.t[b]=c;}};Jrb.Xl=function Xl(a,b,c,d){var e,f,g;if(c==0){a.u[b]&=Lub;a.u[b]|=c<<18;}else {if(d>=32)return;if(d==-1){g=-1;for(f=0;f<a.q;f++)f!=b&&c==(a.u[f]&_tb)>>18&&g<((a.u[f]&_tb)>>18!=1&&(a.u[f]&_tb)>>18!=2?-1:(a.u[f]&Cub)>>20)&&(g=(a.u[f]&_tb)>>18!=1&&(a.u[f]&_tb)>>18!=2?-1:(a.u[f]&Cub)>>20);for(e=0;e<a.r;e++)c==(a.F[e]&768)>>8&&g<((a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&Dub)>>10)&&(g=(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&Dub)>>10);d=g+1;if(d>=32)return;}a.u[b]&=Lub;a.u[b]|=c<<18|d<<20;}a.T&=7;};Jrb.Yl=function Yl(a,b,c){a.v==null&&(a.v=tX(Drb.NY,lub,7,a.M,0,2));Hrb.Nmb(c,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));a.v[b]=c;a.T=0;a.K=true;};Jrb.Zl=function Zl(a,b,c,d){var e;if(c==null){a.v!=null&&(a.v[b]=null);return;}if(c.length==1&&!d){e=c[0];a.C[b]!=e&&Jrb.Tj(a,b,e,0,-1,0);a.v!=null&&(a.v[b]=null);return;}a.v==null&&(a.v=tX(Drb.NY,lub,7,a.M,0,2));a.v[b]=c;d&&(a.B[b]=G3(a.B[b],1));a.T=0;a.K=true;};Jrb.$l=function $l(a,b,c,d){a.w[b]=d?-c:c;};Jrb._l=function _l(a,b,c){c?a.u[b]|=oub:a.u[b]&=Jub;};Jrb.am=function am(a,b,c){a.A[b]=c;a.T&=a.C[b]==1?0:7;};Jrb.bm=function bm(a,b,c,d){a.u[b]&=-8;c!=3&&(a.u[b]&=Kub);a.u[b]|=c;d&&(a.u[b]|=4);};Jrb.cm=function cm(a,b,c,d){d?a.B[b]=G3(a.B[b],c):a.B[b]=j3(a.B[b],F3(c));a.T=0;a.K=true;};Jrb.dm=function dm(a,b,c){a.u[b]&=-49;a.u[b]|=c;a.T&=7;};Jrb.em=function em(a,b,c){c?a.u[b]|=512:a.u[b]&=-513;};Jrb.fm=function fm(a,b,c){a.u[b]&=-67108865;c&&(a.u[b]|=Hub);};Jrb.gm=function gm(a,b,c){a.J[b].a=c;a.T&=7;};Jrb.hm=function hm(a,b,c){a.J[b].b=c;a.T&=7;};Jrb.im=function im(a,b,c){a.J[b].c=c;a.T&=7;};Jrb.jm=function jm(a,b,c){if(c>=0&&c<=190){if(c==151||c==152){a.C[b]=1;a.A[b]=c-149;}else {a.C[b]=c;a.A[b]=0;}a.u[b]&=-2013265921;a.T=0;}};Jrb.km=function km(a,b,c,d){a.D[b][c]=d;a.T=0;};Jrb.lm=function lm(a,b,c){c?a.F[b]|=Iub:a.F[b]&=-32769;};Jrb.mm=function mm(a,b,c){a.F[b]&=-49;a.F[b]|=c<<4;};Jrb.nm=function nm(a,b,c,d){var e,f,g;if(c==0){a.F[b]&=-32513;a.F[b]|=c<<8;}else {if(d>=32)return;if(d==-1){g=-1;for(f=0;f<a.q;f++)c==(a.u[f]&_tb)>>18&&g<((a.u[f]&_tb)>>18!=1&&(a.u[f]&_tb)>>18!=2?-1:(a.u[f]&Cub)>>20)&&(g=(a.u[f]&_tb)>>18!=1&&(a.u[f]&_tb)>>18!=2?-1:(a.u[f]&Cub)>>20);for(e=0;e<a.r;e++)e!=b&&c==(a.F[e]&768)>>8&&g<((a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&Dub)>>10)&&(g=(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&Dub)>>10);d=g+1;if(d>=32)return;}a.F[b]&=-32513;a.F[b]|=c<<8|d<<10;}a.T&=7;};Jrb.om=function om(a,b,c){c?a.F[b]|=Fsb:a.F[b]&=-65537;};Jrb.pm=function pm(a,b,c){a.H[b]=c==1?1:c==2?2:c==3?4:16;a.T=0;};Jrb.qm=function qm(a,b,c,d){a.F[b]&=-131080;a.F[b]|=c;d&&(a.F[b]|=4);};Jrb.rm=function rm(a,b){a.F[b]|=oub;};Jrb.sm=function sm(a,b,c,d){d?a.G[b]|=c:a.G[b]&=~c;a.T=0;d&&c!=0&&(a.K=true);};Jrb.tm=function tm(a,b,c){a.H[b]=c;a.T=0;};Jrb.um=function um(a,b){a.I=b;};Jrb.vm=function vm(a,b){if(a.K!=b){a.K=b;b||Jrb.Ll(a);a.T=0;}};Jrb.wm=function wm(a,b){a.T=b;};Jrb.xm=function xm(a,b){a.S=b;};Jrb.ym=function ym(a,b){var c,d;a.C=Jrb.Pm(a.C,b);a.s=Jrb.Pm(a.s,b);a.w=Jrb.Pm(a.w,b);d=a.J.length;a.J=Jrb.Rm(a.J,b);for(c=d;c<b;c++)a.J[c]=new Jrb.Gh();a.A=Jrb.Pm(a.A,b);a.u=Jrb.Pm(a.u,b);a.B=Jrb.Qm(a.B,b);a.v!=null&&(a.v=Jrb.Tm(a.v,b));a.t!=null&&(a.t=Jrb.Sm(a.t,b));a.M=b;};Jrb.zm=function zm(a,b){a.D[0]=Jrb.Pm(a.D[0],b);a.D[1]=Jrb.Pm(a.D[1],b);a.H=Jrb.Pm(a.H,b);a.F=Jrb.Pm(a.F,b);a.G=Jrb.Pm(a.G,b);a.N=b;};Jrb.Am=function Am(a,b){a.O=b;};Jrb.Bm=function Bm(a,b){a.P=b;};Jrb.Cm=function Cm(a,b){a.u[b]|=Fsb;};Jrb.Dm=function Dm(a,b){var c,d;d=0;for(c=0;c<a.r;c++)(a.D[0][c]==b||a.D[1][c]==b)&&(d+=Jrb.Qk(a,c));return d;};Jrb.Em=function Em(a){var b,c,d;c=false;d=false;for(b=0;b<a.q;b++){if(a.A[b]!=0){a.A[b]=0;c=true;a.C[b]==1&&(d=true);}}d&&(a.T=0);return c;};Jrb.Fm=function Fm(a,b,c){return Jrb.tl(a,b)||Jrb.tl(a,c)?16:1;};Jrb.Gm=function Gm(a,b,c){var d,e,f,g,h,i;g=a.C[b];a.C[b]=a.C[c];a.C[c]=g;g=a.s[b];a.s[b]=a.s[c];a.s[c]=g;g=a.A[b];a.A[b]=a.A[c];a.A[c]=g;g=a.u[b];a.u[b]=a.u[c];a.u[c]=g;i=a.B[b];a.B[b]=a.B[c];a.B[c]=i;g=a.w[b];a.w[b]=a.w[c];a.w[c]=g;f=a.J[b];a.J[b]=a.J[c];a.J[c]=f;if(a.v!=null){h=a.v[b];a.v[b]=a.v[c];a.v[c]=h;}if(a.t!=null){h=a.t[b];a.t[b]=a.t[c];a.t[c]=h;}for(d=0;d<a.r;d++){for(e=0;e<2;e++){a.D[e][d]==b?a.D[e][d]=c:a.D[e][d]==c&&(a.D[e][d]=b);}}a.T=0;};Jrb.Hm=function Hm(a,b,c){var d;d=a.D[0][b];a.D[0][b]=a.D[0][c];a.D[0][c]=d;d=a.D[1][b];a.D[1][b]=a.D[1][c];a.D[1][c]=d;d=a.H[b];a.H[b]=a.H[c];a.H[c]=d;d=a.F[b];a.F[b]=a.F[c];a.F[c]=d;d=a.G[b];a.G[b]=a.G[c];a.G[c]=d;a.T=0;};Jrb.Im=function Im(a,b,c){var d;for(d=0;d<a.q;d++){a.J[d].a+=b;a.J[d].b+=c;}a.U+=b;a.V+=c;};Jrb.Jm=function Jm(a,b,c){var d,e;e=c&127;d=Jrb._k(a,b);switch(e){case 1:case 8:return d>=1;case 2:return d>=2;case 4:return d>=3;case 32:return d>=4;case 64:return d>=5;case 16:return true;default:return false;}};Jrb.Km=function Km(a,b,c,d){var e,f,g;for(e=0;e<a.q;e++){if(!d||(a.u[e]&512)!=0){g=a.R[e]*b;f=a.Q[e]-c;a.J[e].a=a.U+g*$wnd.Math.sin(f);a.J[e].b=a.V+g*$wnd.Math.cos(f);}}d&&(a.T&=7);};Jrb.Lm=function Lm(a,b,c){var d,e,f;a.U=b;a.V=c;a.Q=tX(Drb.LY,Osb,6,a.q,15,1);a.R=tX(Drb.LY,Osb,6,a.q,15,1);for(d=0;d<a.q;d++){e=b-a.J[d].a;f=c-a.J[d].b;a.R[d]=$wnd.Math.sqrt(e*e+f*f);a.Q[d]=Jrb.Vm(b,c,a.J[d].a,a.J[d].b);}};Jrb.Mm=function Mm(){this.M=this.N=256;Jrb.cl(this);};Jrb.Nm=function Nm(a,b){this.M=$wnd.Math.max(1,a);this.N=$wnd.Math.max(1,b);Jrb.cl(this);};Jrb.Pm=function Pm(a,b){var c;c=tX(Drb.NY,Qsb,6,b,15,1);Erb.Y7(a,0,c,0,$wnd.Math.min(a.length,b));return c;};Jrb.Qm=function Qm(a,b){var c;c=tX(Drb.OY,iub,6,b,14,1);Erb.Y7(a,0,c,0,$wnd.Math.min(a.length,b));return c;};Jrb.Rm=function Rm(a,b){var c,d;c=tX(Drb.eZ,Fub,24,b,0,1);for(d=0;d<a.length;d++)!!a[d]&&(c[d]=new Jrb.Ih(a[d]));return c;};Jrb.Sm=function Sm(a,b){var c,d;c=tX(Drb.JY,ssb,9,b,0,2);for(d=0;d<a.length;d++){if(a[d]!=null){c[d]=tX(Drb.JY,nub,6,a[d].length,15,1);Erb.Y7(a[d],0,c[d],0,a[d].length);}}return c;};Jrb.Tm=function Tm(a,b){var c,d;c=tX(Drb.NY,lub,7,b,0,2);for(d=0;d<a.length;d++){if(a[d]!=null){c[d]=tX(Drb.NY,Qsb,6,a[d].length,15,1);Erb.Y7(a[d],0,c[d],0,a[d].length);}}return c;};Jrb.Um=function Um(a){Jrb.Ej();return a>=0&&a<Jrb.zj.length&&Jrb.zj[a]!=null?Jrb.zj[a]:a>=171&&a<=190?Jrb.xj:Jrb.Bj;};Jrb.Vm=function Vm(a,b,c,d){Jrb.Ej();var e,f,g;f=c-a;g=d-b;if(g!=0){e=$wnd.Math.atan(f/g);g<0&&(f<0?e-=ntb:e+=ntb);}else e=f>0?otb:Wtb;return e;};Jrb.Wm=function Wm(a,b){Jrb.Ej();var c;c=a-b;while(c<Mub)c+=mtb;while(c>ntb)c-=mtb;return c;};Jrb.Xm=function Xm(a,b){Jrb.Ej();var c,d,e,f,g;if((b&256)!=0&&Erb.k7(a,'?'))return 0;for(d=1;d<=128;d++)if(!Erb.k7(a,Aub)&&Erb.l7(a,Jrb.yj[d]))return d;if((b&2)!=0)for(e=129;e<=144;e++)if(Erb.l7(a,Jrb.yj[e]))return e;if((b&4)!=0)for(f=146;f<=148;f++)if(Erb.l7(a,Jrb.yj[f]))return f;if((b&1)!=0)for(g=151;g<=152;g++)if(Erb.l7(a,Jrb.yj[g]))return g;if((b&32)!=0)if(Erb.l7(a,Jrb.yj[153]))return 153;if((b&8)!=0)if(Erb.l7(a,Jrb.yj[154]))return 154;if((b&16)!=0)if(Erb.l7(a,Jrb.yj[145]))return 145;if((b&128)!=0)if(Erb.l7(a,Jrb.yj[159]))return 159;if((b&64)!=0)for(c=171;c<=190;c++)if(Erb.l7(a,Jrb.yj[c]))return c;return 0;};Jrb.Ym=function Ym(a){Jrb.Ej();switch(a){case 7:case 8:case 9:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:return true;}return false;};Jrb.Zm=function Zm(a){Jrb.Ej();if(a==1||a==6)return false;if(Jrb.Ym(a))return false;if(a==2||a==10||a==18||a==36||a==54)return false;if(a>103)return false;return true;};Jrb.$m=function $m(a){return a>=3&&a<=4||a>=11&&a<=13||a>=19&&a<=31||a>=37&&a<=51||a>=55&&a<=84||a>=87&&a<=112;};Jrb._m=function _m(a){return a==1||a>=5&&a<=9||a>=14&&a<=17||a>=32&&a<=35||a>=52&&a<=53;};Jrb.an=function an(a){return a>=21&&a<=30||a>=39&&a<=48||a==57||a>=72&&a<=80||a==89||a>=104&&a<=112;};Z3(85,1,{85:1,4:1});_.xb=function Om(a){Jrb.ak(this,a);};_.yb=function bn(a){return this.H[a]==8;};_.q=0;_.r=0;_.I=0;_.K=false;_.L=false;_.M=0;_.N=0;_.O=0;_.S=false;_.T=0;_.U=0;_.V=0;Jrb.Dj=24;Drb.qZ=B5(85);Jrb.cn=function cn(a,b,c,d){var e,f,g,h,i,j,k,l,m;Jrb.ou(b,1);d==null&&(d=tX(Drb.NY,Qsb,6,b.q,15,1));h=Jrb.Ml(a,1);i=Jrb.Ml(a,2);m=tX(Drb.c3,itb,6,b.q,16,1);j=tX(Drb.NY,Qsb,6,b.q,15,1);j[0]=c;m[c]=true;d[c]=Jrb.Yj(b,a,c,h,i);g=0;k=0;while(g<=k){for(l=0;l<Jrb.rn(b,j[g]);l++){f=b.i[j[g]][l];if(!m[f]){j[++k]=f;m[f]=true;d[f]=Jrb.Yj(b,a,f,h,i);}}++g;}for(e=0;e<b.r;e++)m[b.D[0][e]]&&Jrb.Zj(b,a,e,h,i,d==null?b.D[0][e]:d[b.D[0][e]],d==null?b.D[1][e]:d[b.D[1][e]],false);Jrb.Ml(a,1);Jrb.Ml(a,2);a.T=0;};Jrb.dn=function dn(a){var b,c,d,e,f,g,h,i,j,k,l,m;a.j=tX(Drb.NY,Qsb,6,a.q,15,1);a.e=tX(Drb.NY,Qsb,6,a.q,15,1);a.i=tX(Drb.NY,lub,7,a.q,0,2);a.k=tX(Drb.NY,lub,7,a.q,0,2);a.n=tX(Drb.NY,lub,7,a.q,0,2);a.o=tX(Drb.NY,Qsb,6,a.f,15,1);j=tX(Drb.NY,Qsb,6,a.q,15,1);for(g=0;g<a.r;g++){++j[a.D[0][g]];++j[a.D[1][g]];}for(d=0;d<a.q;d++){a.i[d]=tX(Drb.NY,Qsb,6,j[d],15,1);a.k[d]=tX(Drb.NY,Qsb,6,j[d],15,1);a.n[d]=tX(Drb.NY,Qsb,6,j[d],15,1);}l=false;for(h=0;h<a.g;h++){m=Jrb.Qk(a,h);if(m==0){l=true;continue;}for(k=0;k<2;k++){c=a.D[k][h];b=a.e[c];a.n[c][b]=m;a.i[c][b]=a.D[1-k][h];a.k[c][b]=h;++a.e[c];++a.j[c];c<a.f&&(m>1?a.o[c]+=m-1:a.H[h]==8&&(a.o[c]=1));}}for(i=a.g;i<a.r;i++){m=Jrb.Qk(a,i);if(m==0){l=true;continue;}for(k=0;k<2;k++){c=a.D[k][i];b=a.e[c];a.n[c][b]=m;a.i[c][b]=a.D[1-k][i];a.k[c][b]=i;++a.e[c];a.D[1-k][i]<a.f&&++a.j[c];}}if(l){b=tX(Drb.NY,Qsb,6,a.q,15,1);for(e=0;e<a.q;e++)b[e]=a.e[e];for(f=0;f<a.r;f++){m=Jrb.Qk(a,f);if(m==0){for(k=0;k<2;k++){c=a.D[k][f];a.n[c][b[c]]=m;a.i[c][b[c]]=a.D[1-k][f];a.k[c][b[c]]=f;++b[c];}}}}};Jrb.en=function en(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w;Jrb.ou(a,1);c&&(b=true);for(i=0;i<a.r;i++){j=Jrb.Qk(a,i);if(j==1||j==2){if(a.s[a.D[0][i]]>0&&a.s[a.D[1][i]]<0){f=a.D[0][i];g=a.D[1][i];}else if(a.s[a.D[0][i]]<0&&a.s[a.D[1][i]]>0){f=a.D[1][i];g=a.D[0][i];}else continue;if(Jrb.tl(a,f)||Jrb.tl(a,g))continue;if(a.C[f]<9&&Jrb.ao(a,f)>3||a.C[g]<9&&Jrb.ao(a,g)>3)continue;l=Jrb.Wn(a,f)!=0;a.s[f]-=1;a.s[g]+=1;if(!l){s=a.H[i];j==1?a.H[i]=2:a.H[i]=4;if(s==129||s==257){w=a.D[0][i];r=Jrb.Vo(a,w,false);if(a.D[0][r]!=w){a.D[1][r]=a.D[0][r];a.D[1][r]=w;}}}a.T=0;}}t=0;p=0;n=0;for(e=0;e<a.q;e++){t+=a.s[e];if(a.s[e]<0&&!Jrb.uo(a,e)){++p;Jrb.ql(a,e)&&(n-=a.s[e]);}}if(!b&&t!=0)throw h3(new Erb._z("molecule's overall charges are not balanced"));Jrb.ou(a,1);u=0;v=c?t+n:n;for(h=0;h<a.q;h++){if(a.s[h]>0){if(!Jrb.to(a,h)&&Jrb.ql(a,h)){k=$wnd.Math.min(Jrb.Wn(a,h),a.s[h]);if(k!=0&&v>=k){t-=k;u-=k;v-=k;a.s[h]-=k;a.T&=1;}}}}q=c?t:u;if(q<0){o=tX(Drb.NY,Qsb,6,p,15,1);p=0;for(f=0;f<a.q;f++){a.s[f]<0&&!Jrb.uo(a,f)&&(o[p++]=(a.C[f]<<22)+f);}Hrb.Nmb(o,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));for(m=o.length-1;q<0&&m>=o.length-p;m--){d=o[m]&Nub;if(Jrb.ql(a,d)){k=$wnd.Math.min(-q,-a.s[d]);t+=k;q+=k;a.s[d]+=k;a.T&=1;}}}return t;};Jrb.fn=function fn(a){var b,c,d,e,f,g,h,i,j,k,l,m,n;for(c=0;c<a.q;c++){n=Jrb.$k(a,c);n+Jrb.Yk(a,c,n,a.C[c])-Jrb.ao(a,c)<=0&&!(a.s[c]==0&&(a.C[c]==5||(e=a.C[c],e==7||e==15||e==33)||(d=a.C[c],d==8||d==16||d==34||d==52||d==84)))&&(a.B[c]=j3(a.B[c],-6145));}if(a.S)return false;g=false;for(b=0;b<a.f;b++){h=a.e[b]-a.j[b];if(h>0){if(t3(j3(a.B[b],Vtb),0)){k=t3(j3(a.B[b],utb),896)?3:t3(j3(a.B[b],utb),384)?2:t3(j3(a.B[b],128),128)?1:0;i=(n=Jrb.$k(a,b),n+Jrb.Yk(a,b,n,a.C[b])-Jrb.ao(a,b));a.s[b]==0&&t3(j3(a.B[b],wtb),0)&&a.C[b]!=6&&++i;l=h;l>3-k&&(l=3-k);l>i+h-k&&(l=i+h-k);if(l>0){m=k==0?0:H3(j3(a.B[b],utb),l);m=G3(m,Drb.r3((l==3?7:h==2?3:1)<<7));a.B[b]=j3(a.B[b],-1921);a.B[b]=G3(a.B[b],j3(utb,m));}}for(j=a.j[b];j<a.e[b];j++){f=a.k[b][j];if(a.H[f]==1){a.C[a.i[b][j]]=-1;a.H[f]=512;g=true;}}}}g&&Jrb.Xj(a);return g;};Jrb.gn=function gn(a,b){var c,d,e,f;if(a.o[b]==2&&a.j[b]==2&&a.n[b][0]==2){for(e=0;e<2;e++){c=Jrb.mn(a,b,a.i[b][e]);if(c!=-1){for(f=0;f<a.j[c];f++){d=a.k[c][f];(a.H[d]==257||a.H[d]==129)&&a.D[0][d]==c&&(a.H[a.k[c][f]]=1);}}}return;}if(a.o[b]==0||a.C[b]>=15){for(e=0;e<a.e[b];e++){d=a.k[b][e];(a.H[d]==257||a.H[d]==129)&&a.D[0][d]==b&&a.D[0][d]==b&&(a.H[d]=1);}}};Jrb.hn=function hn(a,b,c,d,e){var f,g,h,i,j,k,l,m;Jrb.ou(a,d?7:1);b.v=null;a.K&&Jrb.vm(b,true);i=c.length;e==null&&(e=tX(Drb.NY,Qsb,6,i,15,1));b.q=0;for(f=0;f<i;f++){if(c[f]){e[f]=Jrb.Yj(a,b,f,0,0);if(a.C[f]==1){m=Jrb.ao(a,f);if(m>1){for(l=0;l<a.e[f];l++){if(!c[a.i[f][l]]){Jrb.Ql(b,e[f],m);break;}}}}}else {e[f]=-1;}}k=tX(Drb.NY,Qsb,6,a.r,15,1);Krb.Meb(k,k.length,-1);b.r=0;for(j=0;j<a.r;j++){g=a.D[0][j];h=a.D[1][j];if(g<i&&h<i){if(c[g]&&c[h])k[j]=Jrb.Zj(a,b,j,0,0,e==null?a.D[0][j]:e[a.D[0][j]],e==null?a.D[1][j]:e[a.D[1][j]],d);else if(a.s[g]!=0&&a.s[h]!=0&&a.s[g]<0^a.s[h]<0){c[g]&&(b.s[e[g]]+=a.s[g]<0?1:-1);c[h]&&(b.s[e[h]]+=a.s[h]<0?1:-1);}}}Jrb.ak(a,b);!!a.d&&(b.T=0);b.T=0;Jrb.Ml(b,1);Jrb.Ml(b,2);b.q!=i&&Jrb.vm(b,true);Jrb.Yo(a,b,i,e,k);Jrb.Xo(a,b,i,e);d&&Jrb.de(new Jrb.qe(b),null,false);};Jrb.jn=function jn(a,b,c,d,e){var f,g,h,i,j,k,l,m;Jrb.ou(a,d?7:1);b.v=null;a.K&&Jrb.vm(b,true);e==null&&(e=tX(Drb.NY,Qsb,6,a.q,15,1));b.q=0;for(f=0;f<a.q;f++){e[f]=-1;for(k=0;k<a.j[f];k++){if(c[a.k[f][k]]){e[f]=Jrb.Yj(a,b,f,0,0);if(a.C[f]==1){m=Jrb.ao(a,f);if(m>1){for(l=0;l<a.e[f];l++){if(!c[a.k[f][l]]){Jrb.Ql(b,e[f],m);break;}}}}break;}}}j=tX(Drb.NY,Qsb,6,a.r,15,1);b.r=0;for(i=0;i<a.r;i++)if(c[i]){j[i]=Jrb.Zj(a,b,i,0,0,e==null?a.D[0][i]:e[a.D[0][i]],e==null?a.D[1][i]:e[a.D[1][i]],d);}else {j[i]=-1;g=a.D[0][i];h=a.D[1][i];if(e[g]==-1^e[h]==-1){if(a.s[g]!=0&&a.s[h]!=0&&a.s[g]<0^a.s[h]<0){e[g]!=-1&&(b.s[e[g]]+=a.s[g]<0?1:-1);e[h]!=-1&&(b.s[e[h]]+=a.s[h]<0?1:-1);}}}Jrb.ak(a,b);!!a.d&&(b.T=0);b.T=0;Jrb.Ml(b,1);Jrb.Ml(b,2);b.q!=a.q&&Jrb.vm(b,true);Jrb.Yo(a,b,a.q,e,j);Jrb.Xo(a,b,a.q,e);d&&Jrb.de(new Jrb.qe(b),null,false);return e;};Jrb.kn=function kn(a,b){var c,d,e,f,g,h,i,j,k,l;if((b&~a.T)==0)return;if((a.T&1)==0){Jrb.so(a);Jrb.dn(a);a.T|=1;if(a.K&&!Jrb.el(a)&&Jrb.fn(a)){Jrb.so(a);Jrb.dn(a);a.T|=1;}}if((b&~a.T)==0)return;if((a.T&-7)!=0){for(d=0;d<a.f;d++)a.u[d]&=-15369;for(f=0;f<a.g;f++)a.F[f]&=-705;if((b&4)==0){Jrb.pn(a,1);a.T|=2;return;}Jrb.pn(a,7);for(e=0;e<a.f;e++){for(k=0;k<a.j[e];k++){i=a.k[e][k];if(i<a.g&&Jrb.Gr(a.p,i))continue;h=a.i[e][k];for(l=0;l<a.j[h];l++){if(a.k[h][l]==i)continue;a.n[h][l]>1&&(a.C[a.i[h][l]]==6?a.u[e]|=qtb:!Jrb.zo(a,a.k[h][l])&&Jrb.ql(a,a.i[h][l])&&(a.u[e]|=Utb));}}}while(true){j=false;for(c=0;c<a.f;c++){if(a.o[c]>0&&(a.u[c]&Utb)!=0&&!Jrb.Fr(a.p,c)){for(k=0;k<a.j[c];k++){if(a.n[c][k]>1){h=a.i[c][k];i=a.k[c][k];for(l=0;l<a.j[h];l++){if(a.k[h][l]!=i){g=a.i[h][l];if((a.u[g]&Utb)==0){a.u[g]|=Utb;j=true;}}}}}}}if(!j)break;}}a.T|=6;};Jrb.ln=function ln(a,b){var c,d,e,f,g;c=-1;if(a.o[b]==1){for(f=0;f<a.j[b];f++){if(a.n[b][f]==2){d=a.i[b][f];if(a.j[d]==2&&a.o[d]==2){for(g=0;g<2;g++){e=a.i[d][g];if(e!=b&&a.o[e]==1){c=d;break;}}}break;}}}return c;};Jrb.mn=function mn(a,b,c){var d,e;d=b;while(a.j[c]==2&&a.o[c]==2&&c!=d){e=c;c=a.i[c][0]==b?a.i[c][1]:a.i[c][0];b=e;}return c==d?-1:c;};Jrb.nn=function nn(a,b){var c;if(a.j[b]==3&&b<a.f&&Jrb.Fr(a.p,b)&&(!!a.p&&b<a.f?Jrb.ur(a.p,b):0)>=5)for(c=0;c<a.j[b];c++)if(Jrb.Ao(a,a.k[b][c]))return a.k[b][c];return  -1;};Jrb.on=function on(a,b,c,d,e){var f,g,h,i,j,k;Jrb.ou(a,7);if((a.u[b]&Oub)==0||c&&!(b<a.f&&Jrb.Fr(a.p,b)))return;i=tX(Drb.NY,Qsb,6,a.f,15,1);i[0]=b;d[b]=true;h=0;j=0;while(h<=j){for(k=0;k<a.j[i[h]];k++){g=a.k[i[h]][k];if(!e[g]&&(a.F[g]&64)!=0&&(!c||g<a.g&&Jrb.Gr(a.p,g))){e[g]=true;f=a.i[i[h]][k];if(!d[f]){d[f]=true;i[++j]=f;}}}++h;}};Jrb.pn=function pn(a,b){var c,d,e,f,g,h,i,j;a.p=new Jrb.Tr(a,b);d=tX(Drb.NY,Qsb,6,a.f,15,1);for(e=0;e<a.g;e++){if(Jrb.wr(a.p,e)!=0){a.F[e]|=64;++d[a.D[0][e]];++d[a.D[1][e]];}}for(c=0;c<a.f;c++){d[c]==2?a.u[c]|=vtb:d[c]==3?a.u[c]|=Vtb:d[c]>3&&(a.u[c]|=Oub);}for(j=0;j<a.p.i.a.length;j++){g=Jrb.yr(a.p,j);i=Jrb.zr(a.p,j);h=g.length;for(f=0;f<h;f++){a.u[g[f]]|=8;a.F[i[f]]|=128;a.H[i[f]]==386&&(a.H[i[f]]=2);}}};Jrb.qn=function qn(a){var b,c,d,e,f,g,h,i;h=tX(Drb.c3,itb,6,a.q,16,1);for(c=0;c<a.q;c++)h[c]=a.C[c]==1&&a.A[c]==0&&a.s[c]==0&&(a.u[c]&Bub)==0&&(a.t==null||a.t[c]==null);i=tX(Drb.c3,itb,6,a.q,16,1);for(g=0;g<a.r;g++){d=a.D[0][g];e=a.D[1][g];if(Jrb.Qk(a,g)!=1){h[d]=false;h[e]=false;continue;}i[d]&&(h[d]=false);i[e]&&(h[e]=false);h[d]&&Jrb.tl(a,e)&&a.C[e]!=13&&(h[d]=false);h[e]&&Jrb.tl(a,d)&&a.C[d]!=13&&(h[e]=false);i[d]=true;i[e]=true;}for(f=0;f<a.r;f++){if(h[a.D[0][f]]&&h[a.D[1][f]]){h[a.D[0][f]]=false;h[a.D[1][f]]=false;}}for(b=0;b<a.q;b++)i[b]||(h[b]=false);return h;};Jrb.rn=function rn(a,b){return a.e[b];};Jrb.sn=function sn(a,b){return a.i[b].length;};Jrb.tn=function tn(a,b){return a.e[b]-Jrb._n(a,b)+Jrb.Wn(a,b);};Jrb.un=function un(a){var b,c;Jrb.ou(a,7);b=0;for(c=0;c<a.p.i.a.length;c++)Jrb.Er(a.p,c)&&++b;return b;};Jrb.vn=function vn(a,b){var c,d;c=0;for(d=0;d<a.j[b];d++)Jrb.ql(a,a.i[b][d])&&!Jrb.ll(a,a.k[b][d])&&++c;return c;};Jrb.wn=function wn(a,b){return a.o[b];};Jrb.xn=function xn(a,b){Jrb.ou(a,7);return a.o[b]==2&&a.j[b]==2?Jrb.To(a,b,false):Jrb.Vo(a,b,false);};Jrb.yn=function yn(a,b){var c;c=a.u[b]&Oub;return c==0?0:c==vtb?2:c==Vtb?3:4;};Jrb.zn=function zn(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o;Jrb.ou(a,7);f=tX(Drb.c3,itb,6,a.g,16,1);l=tX(Drb.c3,itb,6,a.g,16,1);o=tX(Drb.NY,Qsb,6,a.f,15,1);g=0;for(h=1;h<a.j[b];h++){d=a.k[b][h];if((a.F[d]&64)!=0){for(j=0;j<h;j++){e=a.k[b][j];if((a.F[e]&64)!=0){l[d]=true;l[e]=true;n=Jrb.co(a,o,a.i[b][h],a.i[b][j],c-2,null,l);l[d]=false;l[e]=false;if(n!=-1){i=false;m=tX(Drb.NY,Qsb,6,n,15,1);Jrb.eo(a,o,m,n);for(k=0;k<n;k++){if(!f[m[k]]){f[m[k]]=true;i=true;}}i&&++g;}}}}}return g;};Jrb.An=function An(a,b){return !!a.p&&b<a.f?Jrb.ur(a.p,b):0;};Jrb.Bn=function Bn(a,b){if(b){Jrb.ou(a,1);return Jrb.Fk(a,a.f,a.g,Jrb.Dj);}else {return Jrb.Fk(a,a.q,a.r,Jrb.Dj);}};Jrb.Cn=function Cn(a){var b,c,d,e,f,g,h,i;Jrb.ou(a,1);h=tX(Drb.MY,Pub,6,a.f,15,1);d=tX(Drb.NY,Qsb,6,a.f,15,1);for(i=0;i<a.f;i++){d[0]=i;e=tX(Drb.NY,Qsb,6,a.f,15,1);e[i]=1;c=0;f=0;while(c<=f){for(g=0;g<a.j[d[c]];g++){b=a.i[d[c]][g];if(e[b]==0){e[b]=e[d[c]]+1;d[++f]=b;h[i]+=e[b]-1;}}++c;}h[i]/=f;}return h;};Jrb.Dn=function Dn(a,b,c){var d;for(d=0;d<a.i[b].length;d++)if(a.i[b][d]==c)return a.k[b][d];return  -1;};Jrb.En=function En(a,b){return !!a.p&&b<a.g?Jrb.wr(a.p,b):0;};Jrb.Fn=function Fn(a,b,c){return a.i[b][c];};Jrb.Gn=function Gn(a,b){return a.j[b];};Jrb.Hn=function Hn(a,b,c){return a.k[b][c];};Jrb.In=function In(a,b,c){return a.n[b][c];};Jrb.Jn=function Jn(a,b){var c,d;c=0;if(a.K)for(d=0;d<a.j[b];d++)E3(j3(a.B[a.i[b][d]],jtb),0)&&++c;return c;};Jrb.Kn=function Kn(a,b){return a.e[b]-a.j[b];};Jrb.Ln=function Ln(a,b,c,d,e){var f,g,h,i;g=a.e[b];if(a.o[b]!=0||b<a.f&&Jrb.Fr(a.p,b)||a.j[b]<3||g>4)return false;i=tX(Drb.c3,itb,6,4,16,1);for(h=0;h<g;h++){f=3.9269908169872414-d[h];if($wnd.Math.abs(Qub-f%otb)>0.0872664675116539)return false;e[h]=3&IY(f/otb);if(i[e[h]])return false;i[e[h]]=true;if((e[h]&1)==0){if(a.H[a.k[b][c[h]]]!=1)return false;}else {if(!Jrb.Al(a,a.k[b][c[h]],b))return false;}}return i[0]&&i[2];};Jrb.Mn=function Mn(a,b,c,d,e){var f,g,h,i,j,k,l,m;if((!!a.p&&b<a.f?Jrb.ur(a.p,b):0)>24)return 3;f=a.e[b];e==null&&(e=tX(Drb.NY,Qsb,6,f,15,1));if(!Jrb.Ln(a,b,c,d,e))return 3;i=-1;for(j=0;j<f;j++){if((e[j]&1)==1){g=a.H[a.k[b][c[j]]];if(i!=-1&&i!=g)return 3;i=g;}}k=$wnd.Math.abs(e[0]-e[1])==2?1:0;h=e[k]-e[k+1];m=$wnd.Math.abs(h)==3^e[k]<e[k+1];l=f==3||(e[3]&1)==1;return m^l^i==129?1:2;};Jrb.Nn=function Nn(a,b,c){var d,e,f,g;g=g=tX(Drb.c3,itb,6,a.q,16,1);f=Jrb.On(a,b,c,g);e=tX(Drb.NY,Qsb,6,f,15,1);f=0;for(d=0;d<a.q;d++)g[d]&&(e[f++]=d);return e;};Jrb.On=function On(a,b,c,d){var e,f,g,h,i,j,k;Jrb.ou(a,1);d==null&&(d=tX(Drb.c3,itb,6,a.q,16,1));i=tX(Drb.NY,Qsb,6,a.q,15,1);i[0]=b;d[b]=true;g=0;j=0;h=1;while(g<=j){f=c?Jrb.sn(a,i[g]):a.e[i[g]];for(k=0;k<f;k++){e=a.i[i[g]][k];if(e<d.length&&!d[e]){i[++j]=e;d[e]=true;++h;}}++g;}return h;};Jrb.Pn=function Pn(a,b,c,d){var e,f,g,h,i,j,k,l,m;Jrb.ou(a,1);for(f=0;f<a.q;f++)b[f]=-1;j=0;for(e=0;e<a.q;e++){if(b[e]==-1&&(!c||(a.u[e]&oub)!=0)){b[e]=j;k=tX(Drb.NY,Qsb,6,a.q,15,1);k[0]=e;i=0;l=0;while(i<=l){h=d?Jrb.sn(a,k[i]):a.e[k[i]];for(m=0;m<h;m++){g=a.i[k[i]][m];if(b[g]==-1&&(!c||(a.u[g]&oub)!=0)){k[++l]=g;b[g]=j;}}++i;}++j;}}return j;};Jrb.Qn=function Qn(a,b,c,d){var e,f,g,h,i,j,k,l,m;Jrb.ou(a,1);for(f=0;f<a.q;f++)b[f]=-1;j=0;for(e=0;e<a.q;e++){if(b[e]==-1){b[e]=j;k=tX(Drb.NY,Qsb,6,a.q,15,1);k[0]=e;i=0;l=0;while(i<=l){h=d?Jrb.sn(a,k[i]):a.e[k[i]];for(m=0;m<h;m++){g=a.i[k[i]][m];if(b[g]==-1&&!c[a.k[k[i]][m]]){k[++l]=g;b[g]=j;}}++i;}++j;}}return j;};Jrb.Rn=function Rn(a,b){var c;return c=Jrb.$k(a,b),c+Jrb.Yk(a,b,c,a.C[b])-Jrb.ao(a,b);};Jrb.Sn=function Sn(a,b){var c,d,e,f,g,h;f=tX(Drb.NY,Qsb,6,a.q,15,1);for(d=0;d<a.q;d++)f[d]=d;e=a.q;do--e;while(e>=0&&b[e]);for(c=0;c<e;c++){if(b[f[c]]){h=f[c];f[c]=f[e];f[e]=h;g=b[c];b[c]=b[e];b[e]=g;do--e;while(b[e]);}}return f;};Jrb.Tn=function Tn(a){return Jrb.Sn(a,Jrb.qn(a));};Jrb.Un=function Un(a,b,c){var d,e,f;e=Jrb.ao(a,b);e-=Jrb.Yk(a,b,e,a.C[b]);c&&(e-=a.e[b]-a.j[b]);f=Jrb.Um(a.C[b]);if(e<=f[0])return  -1;for(d=1;d<f.length;d++)if(f[d]>=e)return f[d];return e;};Jrb.Vn=function Vn(a){var b,c;if(a.K)return 0;Jrb.ou(a,1);c=0;for(b=0;b<a.f;b++)c+=Jrb.Wn(a,b);return c;};Jrb.Wn=function Wn(a,b){var c,d,e,f,g,h,i;if(a.K&&t3(j3(a.B[b],Vtb),0))return 0;if(!Jrb.ep(a,b))return 0;if(a.C[b]==0||Erb.k7('*',a.t==null?null:a.t[b]==null?null:Erb.i7((f=a.t[b],hsb.Kab(),f))))return a.e[b]==0?1:0;Jrb.ou(a,1);h=0;for(e=0;e<a.e[b];e++)h+=a.n[b][e];if(a.K){c=1;for(d=0;d<a.j[b];d++)a.H[a.k[b][d]]==8&&++c;h+=c>>1;}h-=Jrb.Yk(a,b,h,a.C[b]);g=((a.u[b]&Bub)>>>27)-1;if(g==-1){i=Jrb.Um(a.C[b]);g=i[0];for(d=1;g<h&&d<i.length;d++)g=i[d];}return $wnd.Math.max(0,g-h);};Jrb.Xn=function Xn(a,b){var c,d,e,f,g;if(!a.K||a.v==null||a.v[b]==null)return Jrb.Yn(a,b,a.C[b]);g=0;for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];g=$wnd.Math.max(g,Jrb.Yn(a,b,c));}return g;};Jrb.Yn=function Yn(a,b,c){var d,e,f,g,h;f=Jrb.ao(a,b);d=Jrb.Yk(a,b,f,c);g=((a.u[b]&Bub)>>>27)-1;if(g==-1){h=Jrb.Um(a.C[b]);e=0;while(f>h[e]+d&&e<h.length-1)++e;g=h[e];}return g+d-f;};Jrb.Zn=function Zn(a,b){return a.i[b].length-a.e[b];};Jrb.$n=function $n(a){var b,c,d,e;Jrb.ou(a,1);e=0;for(b=0;b<a.q;b++){d=a.A[b]!=0?a.A[b]:Jrb.Cj[a.C[b]];e+=d+Jrb.Wn(a,b)*Jrb.Cj[1];if(a.C[b]>=171&&a.C[b]<=190){c=a.e[b];c>2&&(e-=(c-2)*Jrb.Cj[1]);}}return e;};Jrb._n=function _n(a,b){var c,d;c=a.j[b];for(d=0;d<a.j[b];d++)a.C[a.i[b][d]]==1&&--c;return c;};Jrb.ao=function ao(a,b){var c,d,e,f,g,h;a.zb(1);g=false;d=false;h=0;for(e=0;e<a.e[b];e++){if(!a.K||t3(j3(a.B[a.i[b][e]],jtb),0)){f=a.n[b][e];h+=f;f>1&&(g=true);c=a.k[b][e];a.H[c]==8&&(d=true);}}d&&!g&&++h;return h;};Jrb.bo=function bo(a,b,c){var d,e,f;e=0;for(f=0;f<a.j[b];f++){d=a.i[b][f];d!=c&&a.j[d]>2&&++e;}return e;};Jrb.co=function co(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p;if(c==d){b[0]=c;return 0;}a.zb(1);k=tX(Drb.NY,Qsb,6,a.q,15,1);j=tX(Drb.NY,Qsb,6,a.q,15,1);p=tX(Drb.NY,Qsb,6,a.q,15,1);j[0]=c;k[c]=1;i=0;l=0;while(i<=l&&k[j[i]]<=e){o=j[i];for(m=0;m<a.e[o];m++){if(g==null||g.length<=a.k[o][m]||!g[a.k[o][m]]){h=a.i[o][m];if(h==d){n=k[o];b[n]=h;b[--n]=o;while(n>0){b[n-1]=p[b[n]];--n;}return k[o];}if(k[h]==0&&(f==null||f.length<=h||!f[h])){j[++l]=h;k[h]=k[o]+1;p[h]=o;}}}++i;}return  -1;};Jrb.eo=function eo(a,b,c,d){var e,f;Jrb.ou(a,1);for(e=0;e<d;e++){for(f=0;f<a.e[b[e]];f++){if(a.i[b[e]][f]==b[e+1]){c[e]=a.k[b[e]][f];break;}}}};Jrb.fo=function fo(a,b,c){var d,e,f,g,h,i;if(b==c)return 0;Jrb.ou(a,1);g=tX(Drb.NY,Qsb,6,a.q,15,1);f=tX(Drb.NY,Qsb,6,a.q,15,1);f[0]=b;g[b]=1;e=0;h=0;while(e<=h){for(i=0;i<a.e[f[e]];i++){d=a.i[f[e]][i];if(d==c)return g[f[e]];if(g[d]==0){f[++h]=d;g[d]=g[f[e]]+1;}}++e;}return  -1;};Jrb.go=function go(a,b,c,d,e){var f,g,h,i,j,k;if(b==c)return 0;a.zb(1);i=tX(Drb.NY,Qsb,6,a.q,15,1);h=tX(Drb.NY,Qsb,6,a.q,15,1);h[0]=b;i[b]=1;g=0;j=0;while(g<=j&&i[h[g]]<=d){for(k=0;k<a.e[h[g]];k++){f=a.i[h[g]][k];if(f==c)return i[h[g]];if(i[f]==0&&(e==null||e.length<=f||!e[f])){h[++j]=f;i[f]=i[h[g]]+1;}}++g;}return  -1;};Jrb.ho=function ho(a,b){return a.e[b]-a.j[b]+Jrb.Wn(a,b);};Jrb.io=function io(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;o=tX(Drb.NY,Qsb,6,16,15,1);c=tX(Drb.LY,Osb,6,16,15,1);f=tX(Drb.LY,Osb,6,2,15,1);d=0;for(k=0;k<2;k++){e=a.D[k][b];for(l=0;l<a.j[e];l++){i=a.k[e][l];if(i==b)continue;if(d==4)return 0;h=a.i[e][l];o[d]=16;a.o[h]!=0&&(o[d]+=(a.u[h]&Oub)!=0?1:4);if((a.F[b]&64)!=0&&(a.F[i]&64)!=0){m=Jrb.Cr(a.p,b,i);m!=-1&&(o[d]+=Jrb.Er(a.p,m)?64:6);}c[d++]=Jrb.Vm(a.J[e].a,a.J[e].b,a.J[h].a,a.J[h].b);}}f[0]=Jrb.Hk(a,a.D[0][b],a.D[1][b]);if(f[0]<0){f[1]=f[0]+ntb;g=false;}else {f[1]=f[0];f[0]=f[1]-ntb;g=true;}n=0;for(j=0;j<d;j++){c[j]>f[0]&&c[j]<f[1]?n-=o[j]:n+=o[j];}return g?-n:n;};Jrb.jo=function jo(a){Jrb.ou(a,7);return a.p;};Jrb.ko=function ko(a){var b,c,d,e,f,g,h,i,j;j=0;Jrb.ou(a,7);for(d=0;d<a.g;d++){if(Jrb.Qk(a,d)==1&&(a.F[d]&64)==0){h=true;for(g=0;g<2;g++){b=a.D[g][d];if(a.j[b]==1){h=false;break;}if(a.C[b]==7&&!(b<a.f&&Jrb.Fr(a.p,b))){c=a.D[1-g][d];for(i=0;i<a.j[c];i++){e=a.i[c][i];f=a.k[c][i];if(f!=d&&Jrb.Qk(a,f)>1&&!(e<a.f&&Jrb.Fr(a.p,e))&&Jrb.ql(a,e)){h=false;break;}}}}h&&!Jrb.Io(a,d)&&++j;}}return j;};Jrb.lo=function lo(a,b,c){var d,e,f,g,h,i;e=0;for(f=0;f<a.j[b];f++){d=a.i[b][f];if(d!=c){g=0;for(h=0;h<a.j[d];h++){i=a.i[d][h];i!=b&&Jrb.zo(a,a.k[d][h])&&a.j[i]>2&&++g;}g==2&&++e;}}return e;};Jrb.mo=function mo(a,b){var c,d,e,f;c=a.e[b];f=tX(Drb.NY,Qsb,6,c,15,1);for(e=0;e<c;e++)f[e]=(a.i[b][e]<<16)+e;Hrb.Nmb(f,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));for(d=0;d<c;d++)f[d]&=Gsb;return f;};Jrb.no=function no(a,b){var c,d;Jrb.ou(a,1);if(a.j[b]==2&&a.n[b][0]==2&&a.n[b][1]==2){for(c=0;c<2;c++)for(d=0;d<a.e[a.i[b][c]];d++)if(Jrb.Al(a,a.k[a.i[b][c]][d],a.i[b][c]))return a.k[a.i[b][c]][d];}else {for(c=0;c<a.e[b];c++)if(Jrb.Al(a,a.k[b][c],b))return a.k[b][c];}return  -1;};Jrb.oo=function oo(a,b,c){if(Jrb.Qk(a,b)!=1)return 0;return 16-a.e[c]+16*$wnd.Math.max(0,$wnd.Math.min(9,!!a.p&&b<a.g?Jrb.wr(a.p,b):0)-2)+((a.H[b]&384)==0||a.D[0][b]!=c?Iub:0)+((a.u[c]&3)==0?qtb:0)+(a.C[c]==1?Vtb:0)+(a.e[c]==1?vtb:0)+((a.F[b]&64)!=0?0:512)+(a.C[c]!=6?256:0);};Jrb.po=function po(a,b,c,d,e,f){var g,h,i,j,k;Jrb.ou(a,1);if(e){Jrb.Wj(e);e.d=null;e.K=false;}i=tX(Drb.NY,Qsb,6,a.q,15,1);d==null?d=tX(Drb.c3,itb,6,a.q,16,1):Krb.Qeb(d,d.length);i[0]=b;i[1]=c;d[b]=true;d[c]=true;h=1;j=1;while(h<=j){for(k=0;k<a.e[i[h]];k++){g=a.i[i[h]][k];if(g==b){if(h!=1)return  -1;}if(!d[g]){d[g]=true;i[++j]=g;}}++h;}if(e){f==null&&(f=tX(Drb.NY,Qsb,6,d.length,15,1));Jrb.hn(a,e,d,false,f);Jrb.Tj(e,f[b],0,0,-1,0);}d[b]=false;return j;};Jrb.qo=function qo(a,b,c){var d,e,f,g,h,i;Jrb.ou(a,1);f=tX(Drb.NY,Qsb,6,a.f,15,1);i=tX(Drb.c3,itb,6,a.f,16,1);f[0]=b;f[1]=c;i[b]=true;i[c]=true;e=1;g=1;while(e<=g){for(h=0;h<a.j[f[e]];h++){d=a.i[f[e]][h];if(d==b){if(e!=1)return  -1;}if(!i[d]){i[d]=true;f[++g]=d;}}++e;}return g;};Jrb.ro=function ro(a,b,c){var d,e,f,g,h,i,j,k,l,m,n;if(Jrb.Qk(a,c)!=2&&!(c<a.g&&Jrb.Gr(a.p,c)))return  -1;n=a.F[c]&3;if(n!=1&&n!=2)return  -1;for(i=0;i<2;i++){d=a.D[i][c];e=a.D[1-i][c];m=-1;g=false;for(k=0;k<a.j[d];k++){f=a.i[d][k];f!=e&&(f==b?g=true:m=f);}if(g){l=-1;h=-1;for(j=0;j<a.j[e];j++){f=a.i[e][j];if(f!=d){if(l==-1)l=f;else if(f>l)h=f;else {h=l;l=f;}}}if(a.j[d]==2){if(a.j[e]==2)return n==2?l:-1;return n==2?l:h;}else {if(a.j[e]==2)return n==2^b<m?-1:l;return n==2^b<m?h:l;}}}return  -1;};Jrb.so=function so(a){var b,c,d,e,f,g,h,i,j,k;g=Jrb.qn(a);i=a.q;do--i;while(i>=0&&g[i]);for(b=0;b<i;b++){if(g[b]){Jrb.Gm(a,b,i);k=g[b];g[b]=g[i];g[i]=k;do--i;while(g[i]);}}a.f=i+1;if(a.q==a.f){a.g=a.r;return;}h=tX(Drb.c3,itb,6,a.r,16,1);for(f=0;f<a.r;f++){c=a.D[0][f];d=a.D[1][f];(g[c]||g[d])&&(h[f]=true);}j=a.r;do--j;while(j>=0&&h[j]);for(e=0;e<j;e++){if(h[e]){Jrb.Hm(a,e,j);h[e]=false;do--j;while(h[j]);}}a.g=j+1;};Jrb.to=function to(a,b){var c;for(c=0;c<a.j[b];c++)if(a.s[a.i[b][c]]<0)return true;return false;};Jrb.uo=function uo(a,b){var c;for(c=0;c<a.j[b];c++)if(a.s[a.i[b][c]]>0)return true;return false;};Jrb.vo=function vo(a,b){var c;c=a.C[b];return c==3||c==11||c==19||c==37||c==55||c==87;};Jrb.wo=function wo(a,b){return (a.u[b]&qtb)!=0;};Jrb.xo=function xo(a,b){var c,d,e,f,g,h;Jrb.ou(a,1);for(g=0;g<2;g++){c=a.D[g][b];if(a.C[c]==7){d=a.D[1-g][b];for(h=0;h<a.j[d];h++){e=a.i[d][h];f=a.k[d][h];if((a.C[e]==7||a.C[e]==8||a.C[e]==16)&&Jrb.Qk(a,f)>=2)return true;}}}return false;};Jrb.yo=function yo(a,b){return b<a.f&&Jrb.Fr(a.p,b);};Jrb.zo=function zo(a,b){return b<a.g&&Jrb.Gr(a.p,b);};Jrb.Ao=function Ao(a,b){var c,d,e,f,g,h;if(a.H[b]!=1||b<a.g&&Jrb.Gr(a.p,b)||(a.F[b]&64)!=0&&(!!a.p&&b<a.g?Jrb.wr(a.p,b):0)<7)return false;c=a.D[0][b];if(!(c<a.f&&Jrb.Fr(a.p,c))||(!!a.p&&c<a.f?Jrb.ur(a.p,c):0)<5)return false;d=a.D[1][b];if(!(d<a.f&&Jrb.Fr(a.p,d))||(!!a.p&&d<a.f?Jrb.ur(a.p,d):0)<5)return false;e=Jrb.bo(a,c,d);f=Jrb.bo(a,d,c);if((!!a.p&&c<a.f?Jrb.ur(a.p,c):0)>5&&(!!a.p&&d<a.f?Jrb.ur(a.p,d):0)>5)return e+f>2;g=Jrb.lo(a,c,d);h=Jrb.lo(a,d,c);if(e==2&&h>=1)return true;if(f==2&&g>=1)return true;if(g==2&&(f>=1||h>=1))return true;if(h==2&&(e>=1||g>=1))return true;return false;};Jrb.Bo=function Bo(a,b){return a.o[b]==2&&a.j[b]==2&&a.n[b][0]==2&&a.n[b][1]==2&&a.C[b]<=7;};Jrb.Co=function Co(a,b){var c;c=a.C[b];return c==8||c==16||c==34||c==52||c==84;};Jrb.Do=function Do(a,b){return b<a.g&&(Jrb.Kr(a.p,b)||a.H[b]==8);};Jrb.Eo=function Eo(a,b){var c;c=a.C[b];return c==4||c==12||c==20||c==38||c==56||c==88;};Jrb.Fo=function Fo(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;if(a.C[b]!=7||a.j[b]==4)return false;if(b<a.f&&Jrb.Fr(a.p,b)||a.o[b]!=0||E3(j3(a.B[b],Ttb),0))return true;if(a.s[b]==1)return false;for(i=0;i<a.j[b];i++)if(Jrb.En(a,a.k[b][i])==3)return false;g=0;for(j=0;j<a.j[b];j++){if(a.n[b][j]==1){d=a.C[a.i[b][j]];(d==8||d==9||d==17)&&++g;}}if(g==0){for(h=0;h<a.j[b];h++){e=a.i[b][h];if(a.o[e]!=0){if(e<a.f&&Jrb.Fr(a.p,e)){if((!!a.p&&e<a.f?Jrb.ur(a.p,e):0)>=5){p=0;for(m=0;m<a.j[e];m++){o=a.i[e][m];o!=b&&Jrb._n(a,o)>=3&&++p;}n=Jrb._n(a,b);if(p==2&&n>=2||p==1&&n==3)continue;}return !c||!Jrb.Ko(a,b);}for(l=0;l<a.j[e];l++){if(a.n[e][l]==2||Jrb.zo(a,a.k[e][l]))return !c||!Jrb.Ko(a,b);}}}}if(g<2){for(h=0;h<a.j[b];h++){e=a.i[b][h];k=false;f=false;for(l=0;l<a.j[e];l++){if(a.i[e][l]!=b){a.n[e][l]!=1&&(a.C[a.i[e][l]]==7||a.C[a.i[e][l]]==8||a.C[a.i[e][l]]==16)&&(k=true);a.n[e][l]==1&&a.C[a.i[e][l]]==7&&(f=true);}}if(k&&(!f||g==0))return !c||!Jrb.Ko(a,b);}}return false;};Jrb.Go=function Go(a,b){var c;c=a.C[b];return c==9||c==17||c==35||c==53||c==85;};Jrb.Ho=function Ho(a,b){var c;c=a.C[b];return c==7||c==15||c==33;};Jrb.Io=function Io(a,b){var c,d,e,f,g,h;if(Jrb.Qk(a,b)!=1)return false;for(f=0;f<2;f++){c=a.D[f][b];h=a.D[1-f][b];while(a.o[c]==2&&a.j[c]==2&&a.C[c]<10){for(g=0;g<2;g++){d=a.i[c][g];if(d!=h){if(a.j[d]==1)return true;e=a.k[c][g];if(Jrb.Qk(a,e)==1&&e<b)return true;h=c;c=d;break;}}}if(a.j[c]==1)return true;}return false;};Jrb.Jo=function Jo(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p,q,r,s;m=tX(Drb.c3,itb,6,a.g,16,1);m[e]=true;n=tX(Drb.NY,Qsb,6,11,15,1);o=Jrb.co(a,n,d,b,10,null,m);if(o==-1)return false;g=1;while(!Jrb.Hr(a.p,c,n[g]))++g;f=o-g;h=n[g];s=Jrb.Br(a.p,c);if(s==6&&f==2&&g==3){if(Jrb.yn(a,n[1])>=3){r=Jrb.yr(a.p,c);for(l=0;l<6;l++){if(b==r[l]){q=Jrb.Sr(a.p,c,h==r[Jrb.Sr(a.p,c,l+2)]?l-2:l+2);p=r[q];k=a.u[p]&Oub;if((k==0?0:k==vtb?2:k==Vtb?3:4)>=3&&Jrb.go(a,n[1],p,2,null)==2)return true;break;}}}}i=a.o[h]==1||h<a.f&&Jrb.Fr(a.p,h)||Jrb.Fo(a,h,false);j=!i&&a.C[h]==7&&a.s[h]!=1;if(f==1)return !i&&!j&&s<=4&&g<=3;switch(s){case 4:if(!i&&!j&&g<=4)return true;break;case 5:if(j){if(g<=3)return true;}else if(!i){if(g<=4)return true;}break;case 6:if(f==2){if(i){if(g<=4)return true;}else if(!j){if(g<=3)return true;}}else if(f==3){if(i){if(g<=6)return true;}else {if(g<=4)return true;}}break;case 7:if(f==3){if(g<=3)return true;}}return false;};Jrb.Ko=function Ko(a,b){var c,d,e,f,g;if(b<a.f&&Jrb.Fr(a.p,b)||a.o[b]!=0||E3(j3(a.B[b],Ttb),0)||(d=a.u[b]&Oub,(d==0?0:d==vtb?2:d==Vtb?3:4)<3))return false;g=!!a.p&&b<a.f?Jrb.ur(a.p,b):0;if(g>7)return false;f=0;while(f<a.p.i.a.length){if(Jrb.Br(a.p,f)==g&&Jrb.Hr(a.p,f,b))break;++f;}if(f>=vtb&&f==a.p.i.a.length)return false;for(e=0;e<a.j[b];e++){c=a.k[b][e];if(!Jrb.Ir(a.p,f,c)&&Jrb.Jo(a,b,f,a.i[b][e],c))return true;}return false;};Jrb.Lo=function Lo(a,b){return (a.u[b]&Oub)!=0;};Jrb.Mo=function Mo(a,b){return (a.F[b]&64)!=0;};Jrb.No=function No(a,b){return a.C[b]==1&&a.A[b]==0&&a.s[b]==0&&(a.u[b]&Bub)==0&&(a.t==null||a.t[b]==null);};Jrb.Oo=function Oo(a,b){return (a.u[b]&8)!=0;};Jrb.Po=function Po(a,b){return (a.F[b]&128)!=0;};Jrb.Qo=function Qo(a,b){return (a.u[b]&Utb)!=0;};Jrb.Ro=function Ro(a){var b,c,d,e,f,g,h,i,j,k,l;Jrb.ou(a,1);Jrb.So(a);j=false;for(c=0;c<a.f;c++){if(a.C[c]==7&&a.s[c]==0){l=Jrb.ao(a,c);if(l==4){for(k=0;k<a.j[c];k++){h=a.i[c][k];if(a.n[c][k]==1&&a.C[h]==8&&a.j[h]==1&&a.s[h]==0){j=true;++a.s[c];--a.s[h];break;}}}else if(l==5){for(k=0;k<a.j[c];k++){h=a.i[c][k];i=a.k[c][k];if(a.n[c][k]==2&&a.C[h]==8){j=true;++a.s[c];--a.s[h];a.H[i]=1;break;}if(a.n[c][k]==3&&a.C[h]==7){j=true;++a.s[c];--a.s[h];a.H[i]=2;break;}}}}}g=false;for(f=0;f<a.g;f++){for(k=0;k<2;k++){if(Jrb.ql(a,a.D[k][f])){b=a.D[1-k][f];e=a.C[b];if(e==3||e==11||e==19||e==37||e==55||e==87||(d=a.C[b],d==4||d==12||d==20||d==38||d==56||d==88)){if(Jrb.Qk(a,f)==1){++a.s[b];--a.s[a.D[k][f]];a.H[f]=512;g=true;}else if(a.H[f]==16){a.H[f]=512;g=true;}}break;}}}if(g){Jrb.Xj(a);j=true;}j&&(a.T=0);return j;};Jrb.So=function So(a){var b;for(b=0;b<a.g;b++)if(a.H[b]==8)return Jrb.de(new Jrb.qe(a),null,false);return false;};Jrb.To=function To(a,b,c){var d,e,f,g,h,i,j,k;j=-1;e=0;for(h=0;h<2;h++){d=a.i[b][h];for(i=0;i<a.e[d];i++){f=a.i[d][i];if(f!=b){g=a.k[d][i];k=Jrb.oo(a,g,f);if(e<k&&(!c||!(a.H[g]==257||a.H[g]==129))){e=k;j=g;}}}}return j;};Jrb.Uo=function Uo(a,b){var c,d,e,f,g,h,i,j;i=-1;d=0;for(g=0;g<2;g++){c=a.D[g][b];for(h=0;h<a.e[c];h++){e=a.i[c][h];if(e!=a.D[1-g][b]){f=a.k[c][h];j=Jrb.oo(a,f,e);if(d<j){d=j;i=f;}}}}return i;};Jrb.Vo=function Vo(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;d=a.e[b];e=tX(Drb.LY,Osb,6,d,15,1);for(n=0;n<d;n++)e[n]=Jrb.Hk(a,b,a.i[b][n]);for(o=1;o<d;o++){for(p=0;p<o;p++){f=$wnd.Math.abs(Jrb.Wm(e[o],e[p]));if(f<0.08||f>Rub){g=0;h=0;for(q=0;q<d;q++){if(q!=o&&q!=p){g+=$wnd.Math.abs(Zrb.VU(e[o],e[q]));h+=$wnd.Math.abs(Zrb.VU(e[p],e[q]));}}j=g<h?a.k[b][o]:a.k[b][p];if(Jrb.Qk(a,j)==1&&(!c||!(a.H[j]==257||a.H[j]==129)))return j;}}}r=-1;i=0;for(m=0;m<d;m++){k=a.i[b][m];l=a.k[b][m];s=Jrb.oo(a,l,k);if(i<s&&(!c||!(a.H[l]==257||a.H[l]==129))){i=s;r=l;}}return r;};Jrb.Wo=function Wo(a,b){var c,d,e,f;Jrb.ou(a,b?15:1);a.q=a.f;a.r=a.g;for(d=0;d<a.f;d++){if(a.e[d]!=a.j[d]){c=Jrb.Un(a,d,false);a.e[d]=a.j[d];if(c!=-1){f=Jrb.Un(a,d,true);if(c!=f){e=((a.u[d]&Bub)>>>27)-1;(e==-1||e<c)&&Jrb.Ql(a,d,c);}}}}b&&Jrb.cp(a);a.T=0;};Jrb.Xo=function Xo(a,b,c,d){var e,f,g,h;Jrb.ou(b,1);for(f=0;f<c;f++){if(d[f]!=-1&&a.C[f]!=1&&a.e[f]!=Jrb.rn(b,d[f])){e=Jrb.Un(a,f,false);if(e!=-1){h=Jrb.Un(b,d[f],false);if(e!=h){g=Jrb.kk(b,d[f]);(g==-1||g<e)&&Jrb.Ql(b,d[f],e);}}}}};Jrb.Yo=function Yo(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p;for(h=0;h<c;h++){if(d[h]!=-1&&a.e[h]>3&&(a.u[h]&Hub)!=0){p=0;m=-1;l=-1;for(k=0;k<a.e[h];k++){if(e[a.k[h][k]]!=-1)++p;else if(a.n[h][k]==1&&Jrb.zl(a,a.k[h][k])&&a.D[0][a.k[h][k]]==h){m=a.k[h][k];l=a.i[h][k];}}if(m!=-1&&p>=3){f=Jrb.Vm(a.J[h].a,a.J[h].b,a.J[l].a,a.J[l].b);n=10;o=-1;for(j=0;j<a.e[h];j++){if(a.n[h][j]==1&&(!Jrb.zl(a,a.k[h][j])||a.D[0][a.k[h][j]]==h)&&e[a.k[h][j]]!=-1){g=$wnd.Math.abs(Jrb.Wm(f,Jrb.Hk(a,h,a.i[h][j])));if(n>g){n=g;o=a.k[h][j];}}}if(o!=-1){i=e[o];Jrb.tm(b,i,a.H[o]==257?129:257);if(a.D[0][o]!=h){Jrb.km(b,1,i,d[a.D[0][o]]);Jrb.km(b,0,i,d[h]);}}}}}};Jrb.Zo=function Zo(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;if(a.j[b]!=2||a.n[b][0]!=2||a.n[b][1]!=2||a.j[a.i[b][0]]<2||a.j[a.i[b][1]]<2||a.o[a.i[b][0]]!=1||a.o[a.i[b][1]]!=1){Jrb.bm(a,b,0,false);return;}v=-1;u=-1;t=-1;q=-1;f=0;for(l=0;l<2;l++){d=a.i[b][l];for(o=0;o<a.e[d];o++){g=a.i[d][o];if(g!=b){h=a.k[d][o];w=Jrb.oo(a,h,g);if(f<w){f=w;u=g;v=h;t=d;q=a.i[b][1-l];}}}}if(u==-1)return;for(m=0;m<2;m++){d=a.i[b][m];for(o=0;o<a.e[d];o++){g=a.i[d][o];h=a.k[d][o];g!=b&&a.D[0][h]==d&&(a.H[h]=1);}}if(a.D[1][v]!=u){a.D[0][v]=a.D[1][v];a.D[1][v]=u;}i=xsb;for(n=0;n<a.j[t];n++){g=a.i[t][n];g!=b&&i>g&&(i=g);}r=tX(Drb.NY,Qsb,6,2,15,1);s=0;for(k=0;k<a.j[q];k++){g=a.i[q][k];g!=b&&(r[s++]=g);}c=Jrb.Vm(a.J[b].a,a.J[b].b,a.J[q].a,a.J[q].b);e=0;if(s==2){if(r[0]>r[1]){A=r[0];r[0]=r[1];r[1]=A;}j=Jrb.Wm(c,Jrb.Hk(a,q,r[0]));p=Jrb.Wm(c,Jrb.Hk(a,q,r[1]));e=j-p;}else {e=Jrb.Wm(c,Jrb.Hk(a,q,r[0]));}e<0^(a.u[b]&3)==1^i==u?a.H[v]=257:a.H[v]=129;};Jrb.$o=function $o(a,b,c,d){var e,f,g,h,i,j;e=a.e[b];h=tX(Drb.NY,Qsb,6,e,15,1);j=Jrb.Mn(a,b,c,d,h);if(j==3)return false;g=(a.u[b]&3)==j?257:129;for(i=0;i<e;i++){if((h[i]&1)==1){f=a.k[b][c[i]];a.H[f]=g;if(a.D[0][f]!=b){a.D[1][f]=a.D[0][f];a.D[0][f]=b;}}}return true;};Jrb._o=function _o(a,b){a.T|=248&(8|b);};Jrb.ap=function ap(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q;Jrb.gn(a,b);if((a.u[b]&3)==0||(a.u[b]&3)==3)return;if(a.o[b]==2&&a.j[b]==2){Jrb.Zo(a,b);return;}if(a.j[b]<3||a.j[b]>4){Jrb.bm(a,b,0,false);return;}c=a.e[b];o=false;for(g=0;g<c;g++){if(Jrb.Qk(a,a.k[b][g])==1){o=true;break;}}if(!o)return;p=Jrb.mo(a,b);d=tX(Drb.LY,Osb,6,c,15,1);for(h=0;h<c;h++)d[h]=Jrb.Hk(a,a.i[b][p[h]],b);for(i=0;i<c;i++)a.D[0][a.k[b][i]]==b&&Jrb.Qk(a,a.k[b][i])==1&&(a.H[a.k[b][i]]=1);if((!!a.p&&b<a.f?Jrb.ur(a.p,b):0)<=24&&Jrb.$o(a,b,p,d))return;m=Jrb.Vo(a,b,true);if(m==-1)return;if(a.D[0][m]!=b){a.D[1][m]=a.D[0][m];a.D[0][m]=b;}n=-1;for(j=0;j<c;j++){if(m==a.k[b][p[j]]){n=j;break;}}q=BX(nX(Drb.NY,2),lub,7,0,[BX(nX(Drb.NY,1),Qsb,6,15,[2,1,2,1]),BX(nX(Drb.NY,1),Qsb,6,15,[1,2,2,1]),BX(nX(Drb.NY,1),Qsb,6,15,[1,1,2,2]),BX(nX(Drb.NY,1),Qsb,6,15,[2,1,1,2]),BX(nX(Drb.NY,1),Qsb,6,15,[2,2,1,1]),BX(nX(Drb.NY,1),Qsb,6,15,[1,2,1,2])]);for(f=1;f<c;f++)d[f]<d[0]&&(d[f]+=mtb);if(c==3){k=false;switch(n){case 0:k=d[1]<d[2]&&d[2]-d[1]<ntb||d[1]>d[2]&&d[1]-d[2]>ntb;break;case 1:k=d[2]-d[0]>ntb;break;case 2:k=d[1]-d[0]<ntb;}e=(a.u[b]&3)==1^k?257:129;}else {l=0;d[1]<=d[2]&&d[2]<=d[3]?l=0:d[1]<=d[3]&&d[3]<=d[2]?l=1:d[2]<=d[1]&&d[1]<=d[3]?l=2:d[2]<=d[3]&&d[3]<=d[1]?l=3:d[3]<=d[1]&&d[1]<=d[2]?l=4:d[3]<=d[2]&&d[2]<=d[1]&&(l=5);e=(a.u[b]&3)==1^q[l][n]==1?129:257;}a.H[m]=e;};Jrb.bp=function bp(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;if((a.F[b]&3)==0||(a.F[b]&3)==3||!Jrb.Ao(a,b))return;v=-1;t=-1;u=-1;s=-1;e=0;for(l=0;l<2;l++){d=a.D[l][b];for(o=0;o<a.e[d];o++){h=a.k[d][o];if(h!=b&&Jrb.Qk(a,h)==1){g=a.i[d][o];w=Jrb.oo(a,h,g);if(e<w){e=w;t=g;v=h;u=d;s=a.D[1-l][b];}}}}if(t==-1)return;for(m=0;m<2;m++){for(o=0;o<a.e[a.D[m][b]];o++){h=a.k[a.D[m][b]][o];h!=b&&Jrb.Qk(a,h)==1&&(a.H[h]=1);}}if(a.D[1][v]!=t){a.D[0][v]=a.D[1][v];a.D[1][v]=t;}i=xsb;for(n=0;n<a.j[u];n++){g=a.i[u][n];a.k[u][n]!=b&&i>g&&(i=g);}q=tX(Drb.NY,Qsb,6,2,15,1);r=0;for(k=0;k<a.j[s];k++)a.k[s][k]!=b&&(q[r++]=a.i[s][k]);f=Jrb.Vm(a.J[u].a,a.J[u].b,a.J[s].a,a.J[s].b);c=0;if(r==2){if(q[0]>q[1]){A=q[0];q[0]=q[1];q[1]=A;}j=Jrb.Wm(f,Jrb.Hk(a,s,q[0]));p=Jrb.Wm(f,Jrb.Hk(a,s,q[1]));c=j-p;}else {c=Jrb.Wm(f,Jrb.Hk(a,s,q[0]));}c<0^(a.F[b]&3)==2^i==t?a.H[v]=257:a.H[v]=129;};Jrb.cp=function cp(a){var b,c,d,e;Jrb.ou(a,7);for(d=0;d<a.g;d++)(a.H[d]==257||a.H[d]==129)&&(a.H[d]=1);for(b=0;b<a.f;b++)Jrb.ap(a,b);for(e=0;e<a.g;e++)Jrb.bp(a,e);for(c=0;c<a.g;c++)a.H[c]==2&&(a.F[c]&3)==3&&(a.H[c]=386);};Jrb.dp=function dp(b,c){var d,e,f,g,h,i,j,k,l,m;i=tX(Drb.NY,Qsb,6,b.q,15,1);h=Jrb.Pn(b,i,false,c);if(h<=1)return null;j=tX(Drb.NY,Qsb,6,h,15,1);for(e=0;e<b.f;e++)++j[i[e]];l=0;m=j[0];for(k=1;k<h;k++){if(m<j[k]){m=j[k];l=k;}}for(d=0;d<b.q;d++)i[d]!=l&&(b.C[d]=-1);for(g=0;g<b.r;g++)(!c&&b.H[g]==16||i[b.D[0][g]]!=l)&&(b.H[g]=512);f=Jrb.Xj(b);b.T=0;try{Jrb.en(b,true,true);}catch(a){a=g3(a);if(!zY(a,15))throw h3(a);}return f;};Jrb.ep=function ep(a,b){if((a.u[b]&Bub)!=0)return true;if(a.C[b]<=1)return false;return Jrb.vl(a,b)||a.C[b]==13||a.C[b]>=171;};Jrb.fp=function fp(a){var b,c,d,e,f,g,h,i,j,k;f=Jrb.Fk(a,a.q,a.r,Jrb.Dj);g=f*f/16;for(d=1;d<a.q;d++){for(e=0;e<d;e++){i=a.J[e].a-a.J[d].a;j=a.J[e].b-a.J[d].b;k=a.J[e].c-a.J[d].c;if(i*i+j*j+k*k<g)throw h3(new Erb._z('The distance between two atoms is too close.'));}}Jrb.ou(a,1);b=0;for(c=0;c<a.f;c++){if(Jrb.ao(a,c)>(h=Jrb.$k(a,c),h+Jrb.Yk(a,c,h,a.C[c])))throw h3(new Erb._z('atom valence exceeded'));b+=a.s[c];}if(b!=0)throw h3(new Erb._z('unbalanced atom charge'));};Jrb.gp=function gp(a){var b,c;if(!a.K)return;Jrb.ou(a,7);for(b=0;b<a.f;b++){if((a.u[b]&Oub)!=0){t3(j3(a.B[b],120),8)&&(a.B[b]=j3(a.B[b],-9));a.B[b]=j3(a.B[b],-4294967297);}b<a.f&&Jrb.Fr(a.p,b)?a.B[b]=j3(a.B[b],{l:4194297,m:Nub,h:1048571}):E3(j3(a.B[b],2),0)&&(a.B[b]=j3(a.B[b],-5));(E3(j3(a.B[b],Stb),0)||E3(j3(a.B[b],Usb),0)&&t3(j3(a.B[b],Vsb),0))&&t3(j3(a.B[b],120),8)&&(a.B[b]=j3(a.B[b],-9));a.s[b]!=0&&(a.B[b]=j3(a.B[b],-234881025));if(Jrb.ao(a,b)==(c=Jrb.$k(a,b),c+Jrb.Yk(a,b,c,a.C[b]))){a.B[b]=j3(a.B[b],-4063233);a.B[b]=j3(a.B[b],-17042430230529);a.B[b]=j3(a.B[b],-1921);a.B[b]=j3(a.B[b],-114689);}E3(j3(a.B[b],jtb),0)&&a.w[b]!=0&&Jrb.Kl(a,a.w[b]);}};Jrb.hp=function hp(a){var b,c,d;if(!a.K)return;Jrb.ou(a,7);for(b=0;b<a.g;b++){d=p6(a.G[b]&127);if((b<a.g&&(Jrb.Kr(a.p,b)||a.H[b]==8))&(a.G[b]&8)!=0){a.G[b]&=-9;--d;}if(d!=0){c=a.H[b]&127;c==1?a.G[b]|=1:c==2?a.G[b]|=2:c==4?a.G[b]|=4:c==32?a.G[b]|=32:c==64?a.G[b]|=64:c==16?a.G[b]|=16:c==8&&(a.G[b]|=8);}}};Jrb.ip=function ip(a,b,c){var d;d=Jrb.Jm(a,b,c);if(d&&c==386){Jrb.ou(a,7);d=d&(a.F[b]&128)==0;}return d;};Jrb.jp=function jp(){Jrb.Mm.call(this);};Jrb.kp=function kp(a,b){Jrb.Nm.call(this,a,b);};Jrb.lp=function lp(a){Jrb.Nm.call(this,!a?256:a.M,!a?256:a.N);!!a&&Jrb._j(a,this);};Z3(101,85,{101:1,85:1,4:1});_.zb=function mp(a){Jrb.kn(this,a);};_.yb=function np(a){return Jrb.Do(this,a);};_.f=0;_.g=0;Drb.jZ=B5(101);Jrb.op=function op(a,b){return $wnd.Math.pow(10,$wnd.Math.log(2000)*$wnd.Math.LOG10E*a/(b-1)-1);};Jrb.pp=function pp(a,b){var c,d;c=b;d=0;while(b!=0){if(a.d==0){a.f=(a.c[++a.e]&63)<<11;a.d=6;}d|=(Fsb&a.f)>>16-c+b;a.f<<=1;--b;--a.d;}return d;};Jrb.qp=function qp(a,b,c){a.d=6;a.e=c;a.c=b;a.f=(b[a.e]&63)<<11;};Jrb.rp=function rp(a,b){var c,d,e,f;d=b/2|0;e=a>=d;e&&(a-=d);f=b/32;c=f*a/(d-a);return e?-c:c;};Jrb.sp=function sp(a){var b,c,d,e;if(a.g.K){for(b=0;b<a.g.r;b++){d=Jrb.Sk(a.g,b);if((d&31)==0)continue;c=-1;e=0;if((d&16)!=0){c=16;++e;}if((d&64)!=0){c=64;++e;}if((d&32)!=0){c=32;++e;}if((d&4)!=0){c=4;++e;}if((d&2)!=0){c=2;++e;}if((d&8)!=0){c=8;++e;}if((d&1)!=0){c=1;++e;}if(c!=-1){Jrb.tm(a.g,b,c);e==1&&Jrb.sm(a.g,b,31,false);}}}};Jrb.tp=function tp(a,b){var c;return b==null||Hrb.rnb(b).length==0?null:Jrb.vp(a,Hrb.Umb((c=b,hsb.Kab(),c)));};Jrb.up=function up(a,b,c){var d,e;return b==null?null:Jrb.wp(a,Hrb.Umb((e=b,hsb.Kab(),e)),c==null?null:Hrb.Umb((d=c,d)),0);};Jrb.vp=function vp(a,b){var c;if(b==null||b.length==0)return null;for(c=2;c<b.length-2;c++)if(b[c]==32)return Jrb.wp(a,b,b,c+1);return Jrb.wp(a,b,null,0);};Jrb.wp=function wp(a,b,c,d){var e,f,g,h,i;if(b==null)return null;Jrb.qp(a,b,0);e=Jrb.pp(a,4);h=Jrb.pp(a,4);e>8&&(e=h);f=Jrb.pp(a,e);g=Jrb.pp(a,h);i=new Jrb.Hu(f,g);Jrb.zp(a,i,b,c,d);return i;};Jrb.xp=function xp(a,b,c){var d,e,f,g;if(c==null||Hrb.rnb(c).length==0){Jrb.yp(a,b,null,null);return;}d=Erb.o7(c,A7(32));d>0&&d<Hrb.rnb(c).length-1?Jrb.yp(a,b,Hrb.Umb((f=(Hrb.lnb(0,d,Hrb.rnb(c).length),Hrb.rnb(c).substr(0,d)),hsb.Kab(),f)),Hrb.Umb((g=(Hrb.mnb(d+1,Hrb.rnb(c).length+1),Hrb.rnb(c).substr(d+1)),g))):Jrb.yp(a,b,Hrb.Umb((e=c,hsb.Kab(),e)),null);};Jrb.yp=function yp(a,b,c,d){if(c==null||c.length==0){Jrb.Wj(b);b.d=null;return;}Jrb.zp(a,b,c,d,0);};Jrb.zp=function zp(b,c,d,e,f){var g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$,ab,bb,cb,db,eb,fb,gb,hb,ib,jb,kb,lb,mb,nb,ob,pb,qb,rb,sb,tb,ub,vb,wb,xb,yb,zb,Ab,Bb,Cb,Db,Eb,Fb,Gb,Hb,Ib,Jb,Kb,Lb,Mb,Nb,Ob,Pb,Qb,Rb,Sb,Tb,Ub,Vb,Wb,Xb,Yb,Zb,$b,_b,ac,bc,cc,dc,ec,fc,gc,hc,ic,jc,kc,lc,mc,nc,oc,pc,qc,rc,sc,tc,uc,vc,wc,xc,yc,zc,Ac,Bc,Cc,Dc,Ec,Fc;Jrb.Wj(c);c.d=null;if(d==null||0>=d.length)return;b.g=c;zc=8;e!=null&&(f<0||f>=e.length)&&(e=null);Jrb.qp(b,d,0);i=Jrb.pp(b,4);B=Jrb.pp(b,4);if(i>8){zc=i;i=B;}if(i==0){Jrb.vm(b.g,Jrb.pp(b,1)==1);return;}j=Jrb.pp(b,i);k=Jrb.pp(b,B);mc=Jrb.pp(b,i);qc=Jrb.pp(b,i);pc=Jrb.pp(b,i);O=Jrb.pp(b,i);for(o=0;o<j;o++)Jrb.Hj(b.g,6);for(nb=0;nb<mc;nb++)Jrb.jm(b.g,Jrb.pp(b,i),7);for(ob=0;ob<qc;ob++)Jrb.jm(b.g,Jrb.pp(b,i),8);for(zb=0;zb<pc;zb++)Jrb.jm(b.g,Jrb.pp(b,i),Jrb.pp(b,8));for(Kb=0;Kb<O;Kb++)Jrb.Sl(b.g,Jrb.pp(b,i),Jrb.pp(b,4)-8);P=1+k-j;W=Jrb.pp(b,4);A=0;Jrb.gm(b.g,0,0);Jrb.hm(b.g,0,0);Jrb.im(b.g,0,0);X=e!=null&&e[f]>=39;yc=0;Bc=0;Dc=0;Fc=0;S=false;T=false;if(X){if(e.length>2*j-2&&e[2*j-2]==39||e.length>3*j-3&&e[3*j-3]==39){T=true;S=e.length==3*j-3+9;dc=S?3*j-3:2*j-2;w=86*(e[dc+1]-40)+e[dc+2]-40;yc=$wnd.Math.pow(10,w/2000-1);dc+=2;Ac=86*(e[dc+1]-40)+e[dc+2]-40;Bc=$wnd.Math.pow(10,Ac/1500-1);dc+=2;Cc=86*(e[dc+1]-40)+e[dc+2]-40;Dc=$wnd.Math.pow(10,Cc/1500-1);if(S){dc+=2;Ec=86*(e[dc+1]-40)+e[dc+2]-40;Fc=$wnd.Math.pow(10,Ec/1500-1);}}else {S=e.length==3*j-3;}}if(b.Ab()&&S){e=null;X=false;}for(Vb=1;Vb<j;Vb++){$=Jrb.pp(b,W);if($==0){if(X){Jrb.gm(b.g,Vb,Jrb.Ak(b.g,0)+8*(e[Vb*2-2]-83));Jrb.hm(b.g,Vb,Jrb.Bk(b.g,0)+8*(e[Vb*2-1]-83));S&&Jrb.im(b.g,Vb,Jrb.Ck(b.g,0)+8*(e[2*j-3+Vb]-83));}++P;continue;}A+=$-1;if(X){Jrb.gm(b.g,Vb,Jrb.Ak(b.g,A)+e[Vb*2-2]-83);Jrb.hm(b.g,Vb,Jrb.Bk(b.g,A)+e[Vb*2-1]-83);S&&Jrb.im(b.g,Vb,Jrb.Ck(b.g,A)+(e[2*j-3+Vb]-83));}Jrb.Jj(b.g,A,Vb,1);}for($b=0;$b<P;$b++)Jrb.Jj(b.g,Jrb.pp(b,i),Jrb.pp(b,i),1);ec=tX(Drb.c3,itb,6,k,16,1);for(I=0;I<k;I++){J=Jrb.pp(b,2);switch(J){case 0:ec[I]=true;break;case 2:Jrb.tm(b.g,I,2);break;case 3:Jrb.tm(b.g,I,4);}}h=Jrb.pp(b,i);for(_b=0;_b<h;_b++){n=Jrb.pp(b,i);if(zc==8){rc=Jrb.pp(b,2);if(rc==3){Jrb.Xl(b.g,n,1,0);Jrb.bm(b.g,n,1,false);}else {Jrb.bm(b.g,n,rc,false);}}else {rc=Jrb.pp(b,3);switch(rc){case 4:Jrb.bm(b.g,n,1,false);Jrb.Xl(b.g,n,1,Jrb.pp(b,3));break;case 5:Jrb.bm(b.g,n,2,false);Jrb.Xl(b.g,n,1,Jrb.pp(b,3));break;case 6:Jrb.bm(b.g,n,1,false);Jrb.Xl(b.g,n,2,Jrb.pp(b,3));break;case 7:Jrb.bm(b.g,n,2,false);Jrb.Xl(b.g,n,2,Jrb.pp(b,3));break;default:Jrb.bm(b.g,n,rc,false);}}}zc==8&&Jrb.pp(b,1)==0&&(b.g.L=true);g=Jrb.pp(b,B);for(ac=0;ac<g;ac++){D=Jrb.pp(b,B);if(Jrb.Tk(b.g,D)==1){rc=Jrb.pp(b,3);switch(rc){case 4:Jrb.qm(b.g,D,1,false);Jrb.nm(b.g,D,1,Jrb.pp(b,3));break;case 5:Jrb.qm(b.g,D,2,false);Jrb.nm(b.g,D,1,Jrb.pp(b,3));break;case 6:Jrb.qm(b.g,D,1,false);Jrb.nm(b.g,D,2,Jrb.pp(b,3));break;case 7:Jrb.qm(b.g,D,2,false);Jrb.nm(b.g,D,2,Jrb.pp(b,3));break;default:Jrb.qm(b.g,D,rc,false);}}else {Jrb.qm(b.g,D,Jrb.pp(b,2),false);}}Jrb.vm(b.g,Jrb.pp(b,1)==1);m=null;wc=null;oc=0;while(Jrb.pp(b,1)==1){V=oc+Jrb.pp(b,4);switch(V){case 0:nc=Jrb.pp(b,i);for(bc=0;bc<nc;bc++){n=Jrb.pp(b,i);Jrb.cm(b.g,n,Vtb,true);}break;case 1:nc=Jrb.pp(b,i);for(cc=0;cc<nc;cc++){n=Jrb.pp(b,i);kc=Jrb.pp(b,8);Jrb.am(b.g,n,kc);}break;case 2:nc=Jrb.pp(b,B);for(pb=0;pb<nc;pb++){Jrb.pp(b,B);}break;case 3:nc=Jrb.pp(b,i);for(qb=0;qb<nc;qb++){n=Jrb.pp(b,i);Jrb.cm(b.g,n,qtb,true);}break;case 4:nc=Jrb.pp(b,i);for(rb=0;rb<nc;rb++){n=Jrb.pp(b,i);vc=H3(v3(Jrb.pp(b,4)),3);Jrb.cm(b.g,n,vc,true);}break;case 5:nc=Jrb.pp(b,i);for(sb=0;sb<nc;sb++){n=Jrb.pp(b,i);l=H3(v3(Jrb.pp(b,2)),1);Jrb.cm(b.g,n,l,true);}break;case 6:nc=Jrb.pp(b,i);for(tb=0;tb<nc;tb++){n=Jrb.pp(b,i);Jrb.cm(b.g,n,1,true);}break;case 7:nc=Jrb.pp(b,i);for(ub=0;ub<nc;ub++){n=Jrb.pp(b,i);kb=H3(v3(Jrb.pp(b,4)),7);Jrb.cm(b.g,n,kb,true);}break;case 8:nc=Jrb.pp(b,i);for(vb=0;vb<nc;vb++){n=Jrb.pp(b,i);t=Jrb.pp(b,4);r=tX(Drb.NY,Qsb,6,t,15,1);for(fc=0;fc<t;fc++){s=Jrb.pp(b,8);r[fc]=s;}Jrb.Yl(b.g,n,r);}break;case 9:nc=Jrb.pp(b,B);for(wb=0;wb<nc;wb++){D=Jrb.pp(b,B);vc=Jrb.pp(b,2)<<7;Jrb.sm(b.g,D,vc,true);}break;case 10:nc=Jrb.pp(b,B);for(xb=0;xb<nc;xb++){D=Jrb.pp(b,B);L=Jrb.pp(b,5);Jrb.sm(b.g,D,L,true);}break;case 11:nc=Jrb.pp(b,i);for(yb=0;yb<nc;yb++){n=Jrb.pp(b,i);Jrb.cm(b.g,n,Utb,true);}break;case 12:nc=Jrb.pp(b,B);for(Ab=0;Ab<nc;Ab++){D=Jrb.pp(b,B);M=Jrb.pp(b,8)<<9;Jrb.sm(b.g,D,M,true);}break;case 13:nc=Jrb.pp(b,i);for(Bb=0;Bb<nc;Bb++){n=Jrb.pp(b,i);sc=H3(v3(Jrb.pp(b,3)),14);Jrb.cm(b.g,n,sc,true);}break;case 14:nc=Jrb.pp(b,i);for(Cb=0;Cb<nc;Cb++){n=Jrb.pp(b,i);lc=H3(v3(Jrb.pp(b,5)),17);Jrb.cm(b.g,n,lc,true);}break;case 15:case 31:oc+=16;break;case 16:nc=Jrb.pp(b,i);for(Db=0;Db<nc;Db++){n=Jrb.pp(b,i);uc=H3(v3(Jrb.pp(b,3)),22);Jrb.cm(b.g,n,uc,true);}break;case 17:nc=Jrb.pp(b,i);for(Eb=0;Eb<nc;Eb++){n=Jrb.pp(b,i);Jrb.Ql(b.g,n,Jrb.pp(b,4));}break;case 18:nc=Jrb.pp(b,i);jc=Jrb.pp(b,4);for(Fb=0;Fb<nc;Fb++){n=Jrb.pp(b,i);U=Jrb.pp(b,jc);gc=tX(Drb.JY,nub,6,U,15,1);for(fc=0;fc<U;fc++)gc[fc]=Jrb.pp(b,7)<<24>>24;Jrb.Vl(b.g,n,Erb.i7((hc=gc,hsb.Kab(),hc)));}break;case 19:nc=Jrb.pp(b,i);for(Gb=0;Gb<nc;Gb++){n=Jrb.pp(b,i);N=H3(v3(Jrb.pp(b,3)),25);Jrb.cm(b.g,n,N,true);}break;case 20:nc=Jrb.pp(b,B);for(Hb=0;Hb<nc;Hb++){D=Jrb.pp(b,B);uc=Jrb.pp(b,3)<<17;Jrb.sm(b.g,D,uc,true);}break;case 21:nc=Jrb.pp(b,i);for(Ib=0;Ib<nc;Ib++){n=Jrb.pp(b,i);Jrb.dm(b.g,n,Jrb.pp(b,2)<<4);}break;case 22:nc=Jrb.pp(b,i);for(Jb=0;Jb<nc;Jb++){n=Jrb.pp(b,i);Jrb.cm(b.g,n,Ttb,true);}break;case 23:nc=Jrb.pp(b,B);for(Lb=0;Lb<nc;Lb++){D=Jrb.pp(b,B);Jrb.sm(b.g,D,qub,true);}break;case 24:nc=Jrb.pp(b,B);for(Mb=0;Mb<nc;Mb++){D=Jrb.pp(b,B);l=Jrb.pp(b,2)<<21;Jrb.sm(b.g,D,l,true);}break;case 25:for(Nb=0;Nb<j;Nb++)Jrb.pp(b,1)==1&&Jrb.em(b.g,Nb,true);break;case 26:nc=Jrb.pp(b,B);m=tX(Drb.NY,Qsb,6,nc,15,1);for(Ob=0;Ob<nc;Ob++)m[Ob]=Jrb.pp(b,B);break;case 27:nc=Jrb.pp(b,i);for(Pb=0;Pb<nc;Pb++){n=Jrb.pp(b,i);Jrb.cm(b.g,n,jtb,true);}break;case 28:nc=Jrb.pp(b,B);for(Qb=0;Qb<nc;Qb++)Jrb.tm(b.g,Jrb.pp(b,B),16);break;case 29:nc=Jrb.pp(b,i);for(Rb=0;Rb<nc;Rb++){n=Jrb.pp(b,i);jb=H3(v3(Jrb.pp(b,2)),30);Jrb.cm(b.g,n,jb,true);}break;case 30:nc=Jrb.pp(b,i);for(Sb=0;Sb<nc;Sb++){n=Jrb.pp(b,i);uc=H3(v3(Jrb.pp(b,7)),32);Jrb.cm(b.g,n,uc,true);}break;case 32:nc=Jrb.pp(b,i);for(Tb=0;Tb<nc;Tb++){n=Jrb.pp(b,i);xc=H3(v3(Jrb.pp(b,2)),44);Jrb.cm(b.g,n,xc,true);}break;case 33:nc=Jrb.pp(b,i);for(Ub=0;Ub<nc;Ub++){n=Jrb.pp(b,i);eb=H3(v3(Jrb.pp(b,5)),39);Jrb.cm(b.g,n,eb,true);}break;case 34:nc=Jrb.pp(b,i);for(Wb=0;Wb<nc;Wb++){n=Jrb.pp(b,i);Jrb.cm(b.g,n,ttb,true);}break;case 35:nc=Jrb.pp(b,B);for(Xb=0;Xb<nc;Xb++){D=Jrb.pp(b,B);Jrb.sm(b.g,D,uub,true);}break;case 36:nc=Jrb.pp(b,B);for(Yb=0;Yb<nc;Yb++){D=Jrb.pp(b,B);K=Jrb.pp(b,2)<<5;Jrb.sm(b.g,D,K,true);}break;case 37:nc=Jrb.pp(b,B);for(Zb=0;Zb<nc;Zb++){D=Jrb.pp(b,B);K=Jrb.pp(b,1)==0?32:64;Jrb.tm(b.g,D,K);}break;case 38:nc=Jrb.pp(b,i);Q=Jrb.pp(b,3);wc=tX(Drb.NY,Qsb,6,j,15,1);for(mb=0;mb<nc;mb++)wc[Jrb.pp(b,i)]=Jrb.pp(b,Q);}}Jrb.de(new Jrb.qe(b.g),ec,false);if(m!=null)for(F=m,G=0,H=F.length;G<H;++G){D=F[G];Jrb.tm(b.g,D,Jrb.Tk(b.g,D)==2?4:2);}if(e==null&&!b.i&&d.length>b.e+1&&(d[b.e+1]==32||d[b.e+1]==9)){e=d;f=b.e+2;}if(e!=null){try{if(e[f]==33||e[f]==35){Jrb.qp(b,e,f+1);S=Jrb.pp(b,1)==1;T=Jrb.pp(b,1)==1;tc=2*Jrb.pp(b,4);C=1<<tc;hb=0;D=0;for(p=1;p<j;p++){if(D<k&&Jrb.Ik(b.g,1,D)==p){hb=Jrb.Ik(b.g,0,D++);gb=1;}else {hb=0;gb=8;}Y=gb*(Jrb.pp(b,tc)+1-(C>>1));Z=gb*(Jrb.pp(b,tc)+1-(C>>1));Jrb.gm(b.g,p,Jrb.Ak(b.g,hb)+Y);Jrb.hm(b.g,p,Jrb.Bk(b.g,hb)+Z);S&&Jrb.im(b.g,p,Jrb.Ck(b.g,hb)+gb*(Jrb.pp(b,tc)+1-(C>>1)));}if(e[f]==35){lb=0;ib=tX(Drb.NY,Qsb,6,j,15,1);for(q=0;q<j;q++)lb+=ib[q]=Jrb.Wn(b.g,q);for(n=0;n<j;n++){for(mb=0;mb<ib[n];mb++){kb=Jrb.Hj(b.g,1);Jrb.Jj(b.g,n,kb,1);Jrb.gm(b.g,kb,Jrb.Ak(b.g,n)+(Jrb.pp(b,tc)+1-(C>>1)));Jrb.hm(b.g,kb,Jrb.Bk(b.g,n)+(Jrb.pp(b,tc)+1-(C>>1)));S&&Jrb.im(b.g,kb,Jrb.Ck(b.g,n)+(Jrb.pp(b,tc)+1-(C>>1)));wc!=null&&(wc[n]&1<<mb)!=0&&Jrb.em(b.g,kb,true);}}j+=lb;k+=lb;}v=S?1.5:(Jrb.Ej(),Jrb.Ej(),Jrb.Dj);u=Jrb.Fk(b.g,j,k,v);if(T){yc=Jrb.op(Jrb.pp(b,tc),C);Bc=yc*Jrb.rp(Jrb.pp(b,tc),C);Dc=yc*Jrb.rp(Jrb.pp(b,tc),C);S&&(Fc=yc*Jrb.rp(Jrb.pp(b,tc),C));gb=yc/u;for(n=0;n<j;n++){Jrb.gm(b.g,n,Bc+gb*Jrb.Ak(b.g,n));Jrb.hm(b.g,n,Dc+gb*Jrb.Bk(b.g,n));S&&Jrb.im(b.g,n,Fc+gb*Jrb.Ck(b.g,n));}}else {yc=1.5;gb=yc/u;for(n=0;n<j;n++){Jrb.gm(b.g,n,gb*Jrb.Ak(b.g,n));Jrb.hm(b.g,n,gb*Jrb.Bk(b.g,n));S&&Jrb.im(b.g,n,gb*Jrb.Ck(b.g,n));}}}else {S&&!T&&yc==0&&(yc=1.5);if(yc!=0&&b.g.r!=0){u=0;for(D=0;D<b.g.r;D++){ab=Jrb.Ak(b.g,Jrb.Ik(b.g,0,D))-Jrb.Ak(b.g,Jrb.Ik(b.g,1,D));bb=Jrb.Bk(b.g,Jrb.Ik(b.g,0,D))-Jrb.Bk(b.g,Jrb.Ik(b.g,1,D));cb=S?Jrb.Ck(b.g,Jrb.Ik(b.g,0,D))-Jrb.Ck(b.g,Jrb.Ik(b.g,1,D)):0;u+=$wnd.Math.sqrt(ab*ab+bb*bb+cb*cb);}u/=b.g.r;fb=yc/u;for(n=0;n<b.g.q;n++){Jrb.gm(b.g,n,Jrb.Ak(b.g,n)*fb+Bc);Jrb.hm(b.g,n,Jrb.Bk(b.g,n)*fb+Dc);S&&Jrb.im(b.g,n,Jrb.Ck(b.g,n)*fb+Fc);}}}}catch(a){a=g3(a);if(zY(a,15)){db=a;Erb.Jz(db,(Erb.X7(),Erb.W7),'');'Faulty id-coordinates:'+db+' '+Erb.i7((ic=d,hsb.Kab(),ic))+' '+Erb.i7((hc=e,hc));e=null;S=false;}else throw h3(a);}}R=e!=null&&!S;Jrb.sp(b);if(R||b.Ab()){Jrb.ou(b.g,7);for(D=0;D<b.g.g;D++)Jrb.Qk(b.g,D)==2&&!Jrb.Po(b.g,D)&&Jrb.Rk(b.g,D)==0&&Jrb.rm(b.g,D);}Jrb._o(b.g,0);if(!R&&b.Ab()){try{b.Bb(b.g);R=true;}catch(a){a=g3(a);if(zY(a,15)){db=a;Erb.Jz(db,(Erb.X7(),Erb.W7),'');'2D-coordinate creation failed:'+db+' '+Erb.i7((hc=d,hsb.Kab(),hc));}else throw h3(a);}}if(R){Jrb.cp(b.g);Jrb.Bu(b.g);}else S||Jrb._o(b.g,0);};Jrb.Ap=function Ap(a,b){var c,d,e,f,g,h;if(b.length<=0||b[0]<64)return;Jrb.qp(a,b,0);h=Jrb.pp(a,4);e=Jrb.pp(a,1)==1;f=Jrb.pp(a,1)==1;for(c=0;c<a.g.f;c++){g=Jrb.pp(a,h);d=e;e&&f&&(d=Jrb.pp(a,1)==1);Jrb.$l(a.g,c,g,d);}};Jrb.Bp=function Bp(){};Z3(264,1,{},Jrb.Bp);_.Ab=function Cp(){return false;};_.Bb=function Dp(a){throw h3(new Erb._z('Unexpected request to invent coordinates. Check source code logic!'));};_.d=0;_.e=0;_.f=0;_.i=false;Drb.kZ=B5(264);Jrb.Ep=function Ep(a){};Jrb.Fp=function Fp(a){Jrb.Ep(this);this.b=a;};Z3(50,264,{},Jrb.Fp);_.Ab=function Gp(){return this.b;};_.Bb=function Hp(a){var b;b=new Nrb.zy(this.a);b.g=new Krb.Wib(78187493520);Nrb.gy(b,a);};_.a=2;_.b=false;Drb.lZ=B5(50);Jrb.Ip=function Ip(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r;e=b.a;n=b.g;j=E3(j3(Jrb.yk(a.k,e),1),0);f=Jrb.tk(a.k,e);l=f!=null?Jrb.Pp(a,e,f,d):j?'*':Jrb.sk(a.k,e);!j&&f==null&&Jrb.yo(a.k,e)&&(a.j&4)==0&&(Jrb.wn(a.k,e)!=0||Jrb.kk(a.k,e)==-1&&Jrb.zk(a.k,e)==0)&&(l=Hrb.rnb(l).toLowerCase());b.f&&(c.a+='(',c);n!=-1&&Jrb.Mp(a,Jrb.Dn(a.k,b.a,b.g),b.g,c);g=Jrb.mk(a.k,e);if(g==0&&(a.j&1)!=0){h=j3(Jrb.yk(a.k,e),wtb);o3(h,ztb)==0?g=-1:o3(h,ytb)==0&&(g=1);}k=Jrb.wk(a.k,e);m=(a.j&2)!=0?Jrb.vk(a.k,e):0;o=(a.j&1)!=0?Jrb.Vp(a,e,d):null;r=!j&&!Jrb.aq(Jrb.Dk(a.k,e))||f!=null||(Jrb.xk(a.k,e)==1||Jrb.xk(a.k,e)==2)&&(q=Jrb.rk(a.k,e)-1,!(q!=-1&&a.i[q][Jrb.qk(a.k,e)]<=1))&&(Jrb.Dk(a.k,e)!=7||Jrb.mk(a.k,e)>0)||Jrb.yo(a.k,e)&&Jrb.wn(a.k,e)==0&&(a.j&4)==0||g!=0||k!=0||m!=0||Jrb.kk(a.k,e)!=-1||Jrb.zk(a.k,e)!=0||o!=null;r&&(c.a+='[',c);k!=0&&(c.a+=k,c);c.a+=''+l;(Jrb.xk(a.k,e)==1||Jrb.xk(a.k,e)==2)&&(p=Jrb.rk(a.k,e)-1,!(p!=-1&&a.i[p][Jrb.qk(a.k,e)]<=1))&&(Jrb.Dk(a.k,e)!=7||Jrb.mk(a.k,e)>0)&&Erb.P7(c,Jrb.Up(a,e,n));if((a.j&1)==0&&r){i=Jrb.ho(a.k,e);i==1?(c.a+='H',c):i>1&&(c.a+='H'+i,c);}if(g!=0){c.a+=String.fromCharCode(g>0?43:45);$wnd.Math.abs(g)>1&&Erb.M7(c,$wnd.Math.abs(g));}o!=null&&(c.a+=''+o,c);if(m!=0){c.a+=':';c.a+=m;}r&&(c.a+=']',c);Jrb.Np(a,b,c);b.e&&(c.a+=')',c);};Jrb.Jp=function Jp(a,b,c){var d,e,f,g,h;f=true;h=Krb.Ji(a.g,b).a;g=Jrb.Yp(a,h);while(g!=-1){d=Jrb.Fn(a.k,h,g);e=Jrb.Hn(a.k,h,g);g=Jrb.Yp(a,d);Jrb.Lp(a,new Jrb.ht(d,e,h,c&&f,c&&g==-1),++b);h=d;f=false;}};Jrb.Kp=function Kp(a,b,c){var d,e,f,g;for(e=b,f=0,g=e.length;f<g;++f){d=e[f];a.f[d&1073741823]=(d&Sub)!=0^c?2:1;}};Jrb.Lp=function Lp(a,b,c){Krb.Di(a.g,c,b);a.b[b.a]=true;b.g!=-1&&(a.c[Jrb.Dn(a.k,b.a,b.g)]=true);};Jrb.Mp=function Mp(a,b,c,d){var e,f,g,h,i;i=Hrb.rnb(d.a).length;a.f[b]!=0&&Erb.L7(d,a.f[b]==1?47:92);if(a.j==1){e=Jrb.Sk(a.k,b)&127;if(e!=0){(e&1)!=0&&a.f[b]==0&&(d.a+='-',d);if((e&2)!=0){Hrb.rnb(d.a).length!=i&&(d.a+=',',d);d.a+='=';}if((e&4)!=0){Hrb.rnb(d.a).length!=i&&(d.a+=',',d);d.a+='#';}if((e&32)!=0){Hrb.rnb(d.a).length!=i&&(d.a+=',',d);d.a+='$';}if((e&64)!=0){Hrb.rnb(d.a).length!=i&&(d.a+=',',d);d.a+='$';}if((e&8)!=0){Hrb.rnb(d.a).length!=i&&(d.a+=',',d);d.a+=':';}if((e&16)!=0){Hrb.rnb(d.a).length!=i&&(d.a+=',',d);Erb.P7(d,Jrb.tl(a.k,c)?'<-':'->');}}}if(i==Hrb.rnb(d.a).length&&(!Jrb.zo(a.k,b)||(a.j&4)!=0)){f=Jrb.Tk(a.k,b)&127;f==1?Jrb.yo(a.k,Jrb.Ik(a.k,0,b))&&Jrb.yo(a.k,Jrb.Ik(a.k,1,b))&&(a.j&4)==0&&a.f[b]==0&&(d.a+='-',d):f==2?(d.a+='=',d):f==4?(d.a+='#',d):f==32?(d.a+='$',d):f==64?(d.a+='$',d):f==8?(d.a+=':',d):f==16&&Erb.P7(d,Jrb.tl(a.k,c)?'<-':'->');}if(a.j==1){g=i==Hrb.rnb(d.a).length?'':';';h=Jrb.Sk(a.k,b)&384;h==256?(d.a+=g+'@',d):h==128&&(d.a+=g+'!@',d);}};Jrb.Np=function Np(a,b,c){var d,e,f;if(b.c!=null){for(e=0;e<b.c.length;e++){for(f=0;f<Jrb.Gn(a.k,b.a);f++){if(b.c[e]==Jrb.Fn(a.k,b.a,f)){d=Jrb.Hn(a.k,b.a,f);b.d[e]||Jrb.Mp(a,d,b.a,c);a.e[d]>9&&(c.a+='%',c);Erb.M7(c,a.e[d]);}}}}};Jrb.Op=function Op(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O;M=new Krb.Yi();for(r=new Krb.ueb(a.g);r.a<r.c.a.length;){q=Krb.teb(r);if(q.g!=-1){t=Jrb.Dn(a.k,q.a,q.g);if(!Jrb.Ao(a.k,t)&&!Jrb.Po(a.k,t)&&(Jrb.Rk(a.k,t)==1||Jrb.Rk(a.k,t)==2)){K=Krb.Ji(a.g,a.q[q.g]);c=tX(Drb.NY,Qsb,6,Jrb.Gn(a.k,q.a)+Jrb.Gn(a.k,K.a)-2,15,1);B=0;L=false;if(K.g!=-1){c[B++]=K.b;}else {u=-1;N=-1;v=xsb;for(D=0;D<Jrb.Gn(a.k,K.a);D++){l=Jrb.Fn(a.k,K.a,D);if(l!=q.a){if(u==-1){u=D;v=a.q[l];}else {if(v<a.q[l]){N=D;}else {N=u;u=D;}}}}if(N==-1){H=Jrb.Fn(a.k,K.a,u);I=Jrb.Hn(a.k,K.a,u);c[B++]=I|(Jrb.$p(a,K.a,H)?Sub:0);}else {m=Jrb.Fn(a.k,K.a,u);o=Jrb.Hn(a.k,K.a,u);n=Jrb.Fn(a.k,K.a,N);p=Jrb.Hn(a.k,K.a,N);c[B++]=o|(Jrb.$p(a,K.a,m)?Sub:0);c[B++]=p|(Jrb.$p(a,K.a,n)?0:Sub);}}if(Jrb.Gn(a.k,K.a)==3&&K.g!=-1){for(D=0;D<Jrb.Gn(a.k,K.a);D++){l=Jrb.Fn(a.k,K.a,D);if(l!=K.g&&l!=q.a){d=Jrb.Hn(a.k,K.a,D);c[B++]=d|(Jrb.$p(a,K.a,l)?Sub:0);l<K.g&&(L=!L);break;}}}Jrb.Rk(a.k,t)==2&&(L=!L);for(C=0;C<Jrb.Gn(a.k,q.a);C++){i=Jrb.Fn(a.k,q.a,C);if(i!=q.g){A=L;if(Jrb.Gn(a.k,q.a)==3){for(G=0;G<Jrb.Gn(a.k,q.a);G++){l=Jrb.Fn(a.k,q.a,G);if(l!=q.g&&l!=i){l<i&&(A=!A);break;}}}if(Jrb.ol(a.k,t)){w=Jrb.Mf(a.d,t);if(!a.n[w]){a.n[w]=true;a.o[w]=A;}a.o[w]&&(A=!A);}j=Jrb.Dn(a.k,q.a,i);c[B++]=j|(A^Jrb.$p(a,q.a,i)?0:Sub);}}Hrb.Kmb(M.a,c);}}}a.f=tX(Drb.NY,Qsb,6,a.k.g,15,1);M.a.length!=0&&Jrb.Kp(a,Krb.Qi(M,0),false);while(M.a.length!=0){O=M.a.length;for(C=M.a.length-1;C>=0;C--){c=(Hrb.fnb(C,M.a.length),M.a[C]);J=0;F=false;k=false;for(f=c,g=0,h=f.length;g<h;++g){e=f[g];b=e&1073741823;if(a.f[b]!=0){s=(e&Sub)!=0^a.f[b]==2;J==0?F=s:F!=s&&(k=true);++J;}}if(J!=0){c=Krb.Qi(M,C);k||Jrb.Kp(a,c,F);}}O==M.a.length&&Jrb.Kp(a,Krb.Qi(M,0),false);}};Jrb.Pp=function Pp(a,b,c,d){var e,f,g,h,i,j,k;k=Hrb.rnb(d.a).length;0<k?d.a=Erb.x7(d.a,0,0):0>k&&(d.a+=Erb.D7(tX(Drb.KY,iub,6,-k,15,1)));i=Jrb.yo(a.k,b)&&(a.j&4)==0;for(f=c,g=0,h=f.length;g<h;++g){e=f[g];Hrb.rnb(d.a).length!=0&&(d.a+=',',d);j=(Jrb.Ej(),Jrb.yj)[e];Erb.P7(d,i?Hrb.rnb(j).toLowerCase():j);}return d.a;};Jrb.Qp=function Qp(a){var b,c,d,e,f,g,h,i;if(!a.k||a.k.q==0)return '';Jrb.ou(a.k,15);a.d=new Jrb.fg(a.k,129);e=a.d.S;a.o=tX(Drb.c3,itb,6,e+1,16,1);a.n=tX(Drb.c3,itb,6,e+1,16,1);a.i=rX(Drb.NY,[lub,Qsb],[7,6],15,[2,32],2);for(b=0;b<a.k.f;b++){i=Jrb.rk(a.k,b)-1;i!=-1&&++a.i[i][Jrb.qk(a.k,b)];}Jrb.Tp(a);Jrb.Rp(a);Jrb.Op(a);d=new Erb.S7();c=new Erb.S7();f=true;for(h=new Krb.ueb(a.g);h.a<h.c.a.length;){g=Krb.teb(h);g.g==-1&&(f?f=false:(d.a+='.',d));Jrb.Ip(a,g,d,c);}return d.a;};Jrb.Rp=function Rp(a){var b,c,d,e,f,g,h,i,j,k,l;for(l=new Krb.ueb(a.g);l.a<l.c.a.length;){j=Krb.teb(l);c=0;for(f=0;f<Jrb.Gn(a.k,j.a);f++)a.c[Jrb.Hn(a.k,j.a,f)]||++c;if(c!=0){j.c=tX(Drb.NY,Qsb,6,c,15,1);c=0;for(g=0;g<Jrb.Gn(a.k,j.a);g++){if(!a.c[Jrb.Hn(a.k,j.a,g)]){i=Jrb.Fn(a.k,j.a,g);j.c[c++]=a.q[i]<<16|i;}}Hrb.Nmb(j.c,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));for(e=0;e<j.c.length;e++)j.c[e]=Gsb&j.c[e];}}d=tX(Drb.c3,itb,6,a.k.g,16,1);a.e=tX(Drb.NY,Qsb,6,a.k.g,15,1);for(k=new Krb.ueb(a.g);k.a<k.c.a.length;){j=Krb.teb(k);if(j.c!=null){j.d=tX(Drb.c3,itb,6,j.c.length,16,1);for(e=0;e<j.c.length;e++){for(h=0;h<Jrb.Gn(a.k,j.a);h++){if(j.c[e]==Jrb.Fn(a.k,j.a,h)){b=Jrb.Hn(a.k,j.a,h);if(a.c[b]){d[a.e[b]]=false;}else {a.c[b]=true;j.d[e]=true;a.e[b]=1;while(d[a.e[b]])++a.e[b];d[a.e[b]]=true;}}}}}}};Jrb.Sp=function Sp(a){var b,c,d,e;d=-1;e=xsb;for(b=0;b<a.k.f;b++){if(!a.b[b]){c=a.a[b];E3(j3(Jrb.yk(a.k,b),jtb),0)&&(c+=Sub);Jrb.Gn(a.k,b)==0?c+=1056964608:c+=Jrb.Gn(a.k,b)<<24;if(e>c){e=c;d=b;}}}return d;};Jrb.Tp=function Tp(a){var b,c,d,e,f;a.a=a.d.e;a.b=tX(Drb.c3,itb,6,a.k.f,16,1);a.c=tX(Drb.c3,itb,6,a.k.g,16,1);a.g=new Krb.Yi();b=Jrb.Sp(a);while(b!=-1){c=a.g.a.length;Jrb.Lp(a,new Jrb.ht(b,-1,-1,false,false),c);if(Jrb.Gn(a.k,b)!=0){Jrb.Jp(a,c,false);while(c<a.g.a.length-1){while(Jrb.Zp(a,Krb.Ji(a.g,c).a))Jrb.Jp(a,c,true);++c;}}b=Jrb.Sp(a);}a.q=tX(Drb.NY,Qsb,6,a.k.f,15,1);d=0;for(f=new Krb.ueb(a.g);f.a<f.c.a.length;){e=Krb.teb(f);a.q[e.a]=d++;}};Jrb.Up=function Up(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o;i=false;if(Jrb.wn(a.k,b)!=0&&Jrb.Gn(a.k,b)==2&&Jrb.In(a.k,b,0)==2&&Jrb.In(a.k,b,1)==2){for(g=0;g<Jrb.Gn(a.k,b);g++){d=Jrb.Fn(a.k,b,g);if(d!=c&&Jrb.Gn(a.k,d)==2){i=true;break;}}for(f=0;f<Jrb.Gn(a.k,b);f++){d=Jrb.Fn(a.k,b,f);o=0;n=tX(Drb.NY,Qsb,6,3,15,1);for(k=0;k<Jrb.Gn(a.k,d);k++){n[o]=Jrb.Fn(a.k,d,k);n[o]!=b&&++o;}o==2&&a.q[n[0]]<a.q[n[1]]^n[0]<n[1]&&(i=!i);}}else {l=tX(Drb.NY,Qsb,6,4,15,1);m=tX(Drb.NY,Qsb,6,4,15,1);h=0;if(c!=-1){l[h]=c;m[h++]=8*a.q[c];}if(Jrb.Wn(a.k,b)!=0){l[h]=xsb;m[h++]=8*a.q[b];}else if(Jrb.Gn(a.k,b)==3){l[h]=xsb;m[h++]=8*a.q[b];}for(f=0;f<Jrb.Gn(a.k,b);f++){d=Jrb.Fn(a.k,b,f);if(d!=c){l[h]=d;m[h++]=Jrb.Xp(a,b,f);}}i=Jrb._p(l,m);}j=Jrb.xk(a.k,b)==1^i;if(Jrb.hl(a.k,b)){e=Jrb.Nf(a.d,b);if(!a.n[e]){a.n[e]=true;a.o[e]=j;}a.o[e]&&(j=!j);}return j?'@@':'@';};Jrb.Vp=function Vp(a,b,c){var d,e,f,g,h,i,j,k;h=Hrb.rnb(c.a).length;0<h?c.a=Erb.x7(c.a,0,0):0>h&&(c.a+=Erb.D7(tX(Drb.KY,iub,6,-h,15,1)));i=Jrb.yk(a.k,b);e=N3(I3(j3(i,wtb),3));switch(e){case 20971520:c.a+='+0';break;case 25165824:Jrb.mk(a.k,b)==0&&(c.a+='-',c);break;case 12582912:Jrb.mk(a.k,b)==0&&(c.a+='+',c);}d=j3(i,pub);o3(d,2)==0?(c.a+=';a',c):o3(d,4)==0&&(c.a+=';A',c);f=j3(i,utb);o3(f,0)!=0&&(o3(f,1792)==0?(c.a+=';H0',c):o3(f,1664)==0?(c.a+=';H1',c):o3(f,1408)==0?(c.a+=';H2',c):o3(f,896)==0?(c.a+=';H3',c):o3(f,128)==0?(c.a+=';!H0',c):o3(f,384)==0?(c.a+=';!H0;!H1',c):o3(f,1536)==0?(c.a+=';!H2;!H3',c):o3(f,vtb)==0&&(c.a+=';!H3',c));k=j3(i,120);o3(k,8)==0?(c.a+=';!R0',c):o3(k,16)==0?(c.a+=';!R1',c):o3(k,32)==0?(c.a+=';!R2',c):o3(k,64)==0?(c.a+=';!R3',c):o3(k,112)==0?(c.a+=';R0',c):o3(k,104)==0?(c.a+=';R1',c):o3(k,88)==0?(c.a+=';R2',c):o3(k,56)==0&&(c.a+=';R3',c);j=j3(i,Usb);if(o3(j,Vsb)==0)c.a+=';!r'+O3(j);else if(o3(j,Tub)==0)c.a+=';r'+O3(j);else if(o3(j,0)!=0){if(E3(j3(j,_sb),0)){t3(j3(j,Vsb),0)&&(c.a+=';!r0'+O3(j),c);t3(j3(j,Wsb),0)&&(c.a+=';!r3'+O3(j),c);t3(j3(j,Xsb),0)&&(c.a+=';!r4'+O3(j),c);t3(j3(j,Ysb),0)&&(c.a+=';!r5'+O3(j),c);t3(j3(j,Zsb),0)&&(c.a+=';!r6'+O3(j),c);t3(j3(j,$sb),0)&&(c.a+=';!r7'+O3(j),c);}else {c.a+=';';E3(j3(j,Vsb),0)&&(c.a+='r0,'+O3(j),c);E3(j3(j,Wsb),0)&&(c.a+='r3,'+O3(j),c);E3(j3(j,Xsb),0)&&(c.a+='r4,'+O3(j),c);E3(j3(j,Ysb),0)&&(c.a+='r5,'+O3(j),c);E3(j3(j,Zsb),0)&&(c.a+='r6,'+O3(j),c);E3(j3(j,$sb),0)&&(c.a+='r7,'+O3(j),c);Erb.G4(c,Hrb.rnb(c.a).length-1);}}if(o3(j,0)==0){j=I3(j3(i,Stb),22);o3(j,0)!=0&&(c.a+=';r'+O3(j),c);}g=j3(i,Btb);o3(g,3801088)==0&&(c.a+=';D1',c);o3(g,Ctb)==0&&(c.a+=';D2',c);o3(g,Dtb)==0&&(c.a+=';D3',c);o3(g,3145728)==0&&(c.a+=';!D3;!D4',c);o3(g,Etb)==0&&(c.a+=';!D4',c);o3(g,Ftb)==0&&(c.a+=';!D0;!D1',c);o3(g,Gtb)==0&&(c.a+=';!D0;!D1;!D2',c);o3(g,Htb)==0&&(c.a+=';!D0;!D1;!D2;!D3',c);E3(j3(i,Vtb),0)&&Erb.P7(c,';D'+Jrb.Gn(a.k,b));E3(j3(i,qtb),0)&&Erb.P7(c,';!D'+Jrb.Gn(a.k,b));return Hrb.rnb(c.a).length==0?null:c.a;};Jrb.Wp=function Wp(a){a.p==null&&(a.p=Jrb.Qp(a));return a.p;};Jrb.Xp=function Xp(a,b,c){var d,e,f,g,h;d=Jrb.Hn(a.k,b,c);g=Jrb.Fn(a.k,b,c);if(a.e[d]!=0){h=8*a.q[b]+1;e=Krb.Ji(a.g,a.q[b]).c;for(f=0;f<e.length&&g!=e[f];f++)++h;return h;}return 8*a.q[g];};Jrb.Yp=function Yp(a,b){var c,d,e,f,g;c=-1;d=-1;for(f=0;f<Jrb.Gn(a.k,b);f++){e=Jrb.Fn(a.k,b,f);g=(Jrb.In(a.k,b,f)<<24)+a.a[e];if(!a.b[e]&&(c==-1||d<g)){c=f;d=g;}}return c;};Jrb.Zp=function Zp(a,b){var c;for(c=0;c<Jrb.Gn(a.k,b);c++)if(!a.b[Jrb.Fn(a.k,b,c)])return true;return false;};Jrb.$p=function $p(a,b,c){var d,e;d=Krb.Ji(a.g,a.q[b]);if(d.g==c)return false;e=Krb.Ji(a.g,a.q[c]);if(e.g==b)return true;return Jrb.gt(e,b);};Jrb._p=function _p(a,b){var c,d,e;d=false;for(c=1;c<4;c++){for(e=0;e<c;e++){a[e]>a[c]&&(d=!d);b[e]>b[c]&&(d=!d);}}return d;};Jrb.aq=function aq(a){return a>=5&&a<=9||a>=15&&a<=17||a==35||a==53;};Jrb.bq=function bq(a){Jrb.cq.call(this,a,0);};Jrb.cq=function cq(a,b){this.k=a;this.j=b;};Jrb.dq=function dq(a){var b,c,d,e;e=new Erb.S7();for(c=0;c<a.g.a.length;c++){c!=0&&(e.a+='.',e);Erb.P7(e,Jrb.Wp(new Jrb.cq(Krb.Ji(a.g,c),2)));}e.a+='>';for(d=0;d<a.a.a.length;d++){d!=0&&(e.a+='.',e);Erb.P7(e,Jrb.Wp(new Jrb.bq(Krb.Ji(a.a,d))));}e.a+='>';for(b=0;b<a.f.a.length;b++){b!=0&&(e.a+='.',e);Erb.P7(e,Jrb.Wp(new Jrb.cq(Krb.Ji(a.f,b),2)));}return e.a;};Z3(98,1,{},Jrb.bq,Jrb.cq);_.j=0;Drb.mZ=B5(98);Jrb.eq=function eq(a,b){this.b=a;this.a=b;};Z3(3,1,{3:1},Jrb.eq);_.a=0;_.b=0;Drb.nZ=B5(3);Jrb.gq=function gq(){Jrb.gq=_3;Jrb.fq=BX(nX(Drb.nZ,2),ssb,14,0,[null,BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(0,1.007825032),new Jrb.eq(1,2.014101778),new Jrb.eq(2,3.016049268),new Jrb.eq(3,4.027834627),new Jrb.eq(4,5.039542911),new Jrb.eq(5,6.044942608)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(1,3.01602931),new Jrb.eq(2,4.00260325),new Jrb.eq(3,5.012223628),new Jrb.eq(4,6.018888072),new Jrb.eq(5,7.028030527),new Jrb.eq(6,8.033921838),new Jrb.eq(7,9.043820323),new Jrb.eq(8,10.052399713)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(1,4.027182329),new Jrb.eq(2,5.012537796),new Jrb.eq(3,6.015122281),new Jrb.eq(4,7.016004049),new Jrb.eq(5,8.02248667),new Jrb.eq(6,9.026789122),new Jrb.eq(7,10.035480884),new Jrb.eq(8,11.043796166),new Jrb.eq(9,12.05378)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(1,5.04079),new Jrb.eq(2,6.019725804),new Jrb.eq(3,7.016929246),new Jrb.eq(4,8.005305094),new Jrb.eq(5,9.012182135),new Jrb.eq(6,10.01353372),new Jrb.eq(7,11.021657653),new Jrb.eq(8,12.026920631),new Jrb.eq(9,13.036133834),new Jrb.eq(10,14.042815522)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(2,7.029917389),new Jrb.eq(3,8.024606713),new Jrb.eq(4,9.013328806),new Jrb.eq(5,10.012937027),new Jrb.eq(6,11.009305466),new Jrb.eq(7,12.014352109),new Jrb.eq(8,13.017780267),new Jrb.eq(9,14.025404064),new Jrb.eq(10,15.031097291),new Jrb.eq(11,16.039808836),new Jrb.eq(12,17.046931399),new Jrb.eq(13,18.05617),new Jrb.eq(14,19.06373)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(2,8.037675026),new Jrb.eq(3,9.031040087),new Jrb.eq(4,10.01685311),new Jrb.eq(5,11.011433818),new Jrb.eq(6,12),new Jrb.eq(7,13.003354838),new Jrb.eq(8,14.003241988),new Jrb.eq(9,15.010599258),new Jrb.eq(10,16.014701243),new Jrb.eq(11,17.022583712),new Jrb.eq(12,18.026757058),new Jrb.eq(13,19.035248094),new Jrb.eq(14,20.040322395),new Jrb.eq(15,21.04934),new Jrb.eq(16,22.05645)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(3,10.042618),new Jrb.eq(4,11.026796226),new Jrb.eq(5,12.018613202),new Jrb.eq(6,13.005738584),new Jrb.eq(7,14.003074005),new Jrb.eq(8,15.000108898),new Jrb.eq(9,16.006101417),new Jrb.eq(10,17.008449673),new Jrb.eq(11,18.014081827),new Jrb.eq(12,19.017026896),new Jrb.eq(13,20.023367295),new Jrb.eq(14,21.027087574),new Jrb.eq(15,22.034440259),new Jrb.eq(16,23.04051),new Jrb.eq(17,24.0505)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(4,12.034404776),new Jrb.eq(5,13.0248104),new Jrb.eq(6,14.008595285),new Jrb.eq(7,15.003065386),new Jrb.eq(8,15.994914622),new Jrb.eq(9,16.999131501),new Jrb.eq(10,17.999160419),new Jrb.eq(11,19.00357873),new Jrb.eq(12,20.00407615),new Jrb.eq(13,21.008654631),new Jrb.eq(14,22.009967157),new Jrb.eq(15,23.015691325),new Jrb.eq(16,24.020369922),new Jrb.eq(17,25.02914),new Jrb.eq(18,26.03775)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(5,14.03608),new Jrb.eq(6,15.018010856),new Jrb.eq(7,16.01146573),new Jrb.eq(8,17.002095238),new Jrb.eq(9,18.000937667),new Jrb.eq(10,18.998403205),new Jrb.eq(11,19.999981324),new Jrb.eq(12,20.999948921),new Jrb.eq(13,22.00299925),new Jrb.eq(14,23.003574385),new Jrb.eq(15,24.008099371),new Jrb.eq(16,25.012094963),new Jrb.eq(17,26.019633157),new Jrb.eq(18,27.026892316),new Jrb.eq(19,28.03567),new Jrb.eq(20,29.04326)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(6,16.025756907),new Jrb.eq(7,17.017697565),new Jrb.eq(8,18.005697066),new Jrb.eq(9,19.001879839),new Jrb.eq(10,19.992440176),new Jrb.eq(11,20.993846744),new Jrb.eq(12,21.99138551),new Jrb.eq(13,22.994467337),new Jrb.eq(14,23.993615074),new Jrb.eq(15,24.997789899),new Jrb.eq(16,26.000461498),new Jrb.eq(17,27.0076152),new Jrb.eq(18,28.012108072),new Jrb.eq(19,29.019345902),new Jrb.eq(20,30.023872),new Jrb.eq(21,31.03311),new Jrb.eq(22,32.03991)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(7,18.02718),new Jrb.eq(8,19.01387945),new Jrb.eq(9,20.00734826),new Jrb.eq(10,20.997655099),new Jrb.eq(11,21.994436782),new Jrb.eq(12,22.989769675),new Jrb.eq(13,23.990963332),new Jrb.eq(14,24.989954352),new Jrb.eq(15,25.992589898),new Jrb.eq(16,26.994008702),new Jrb.eq(17,27.99889041),new Jrb.eq(18,29.002811301),new Jrb.eq(19,30.009226487),new Jrb.eq(20,31.013595108),new Jrb.eq(21,32.019649792),new Jrb.eq(22,33.027386),new Jrb.eq(23,34.0349),new Jrb.eq(24,35.04418)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(8,20.018862744),new Jrb.eq(9,21.011714174),new Jrb.eq(10,21.999574055),new Jrb.eq(11,22.99412485),new Jrb.eq(12,23.985041898),new Jrb.eq(13,24.985837023),new Jrb.eq(14,25.98259304),new Jrb.eq(15,26.984340742),new Jrb.eq(16,27.983876703),new Jrb.eq(17,28.988554743),new Jrb.eq(18,29.990464529),new Jrb.eq(19,30.996548459),new Jrb.eq(20,31.999145889),new Jrb.eq(21,33.005586975),new Jrb.eq(22,34.00907244),new Jrb.eq(23,35.018669),new Jrb.eq(24,36.02245),new Jrb.eq(25,37.03124)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(8,21.02804),new Jrb.eq(9,22.01952),new Jrb.eq(10,23.0072649),new Jrb.eq(11,23.999940911),new Jrb.eq(12,24.990428555),new Jrb.eq(13,25.986891659),new Jrb.eq(14,26.981538441),new Jrb.eq(15,27.981910184),new Jrb.eq(16,28.980444848),new Jrb.eq(17,29.982960304),new Jrb.eq(18,30.983946023),new Jrb.eq(19,31.988124379),new Jrb.eq(20,32.990869587),new Jrb.eq(21,33.996927255),new Jrb.eq(22,34.99993765),new Jrb.eq(23,36.006351501),new Jrb.eq(24,37.01031),new Jrb.eq(25,38.0169),new Jrb.eq(26,39.0219)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(8,22.03453),new Jrb.eq(9,23.02552),new Jrb.eq(10,24.011545711),new Jrb.eq(11,25.00410664),new Jrb.eq(12,25.992329935),new Jrb.eq(13,26.986704764),new Jrb.eq(14,27.976926533),new Jrb.eq(15,28.976494719),new Jrb.eq(16,29.973770218),new Jrb.eq(17,30.975363275),new Jrb.eq(18,31.974148129),new Jrb.eq(19,32.97800052),new Jrb.eq(20,33.978575745),new Jrb.eq(21,34.984584158),new Jrb.eq(22,35.986687363),new Jrb.eq(23,36.99299599),new Jrb.eq(24,37.99598),new Jrb.eq(25,39.0023),new Jrb.eq(26,40.0058),new Jrb.eq(27,41.0127),new Jrb.eq(28,42.0161)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(9,24.03435),new Jrb.eq(10,25.02026),new Jrb.eq(11,26.01178),new Jrb.eq(12,26.999191645),new Jrb.eq(13,27.99231233),new Jrb.eq(14,28.981801376),new Jrb.eq(15,29.978313807),new Jrb.eq(16,30.973761512),new Jrb.eq(17,31.973907163),new Jrb.eq(18,32.971725281),new Jrb.eq(19,33.973636381),new Jrb.eq(20,34.973314249),new Jrb.eq(21,35.978259824),new Jrb.eq(22,36.979608338),new Jrb.eq(23,37.98447),new Jrb.eq(24,38.98642),new Jrb.eq(25,39.99105),new Jrb.eq(26,40.9948),new Jrb.eq(27,42.00009),new Jrb.eq(28,43.00331),new Jrb.eq(29,44.00988),new Jrb.eq(30,45.01514),new Jrb.eq(31,46.02383)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(10,26.02788),new Jrb.eq(11,27.018795),new Jrb.eq(12,28.004372661),new Jrb.eq(13,28.996608805),new Jrb.eq(14,29.984902954),new Jrb.eq(15,30.979554421),new Jrb.eq(16,31.97207069),new Jrb.eq(17,32.971458497),new Jrb.eq(18,33.967866831),new Jrb.eq(19,34.96903214),new Jrb.eq(20,35.96708088),new Jrb.eq(21,36.971125716),new Jrb.eq(22,37.971163443),new Jrb.eq(23,38.975135275),new Jrb.eq(24,39.97547),new Jrb.eq(25,40.98003),new Jrb.eq(26,41.98149),new Jrb.eq(27,42.9866),new Jrb.eq(28,43.98832),new Jrb.eq(29,44.99482),new Jrb.eq(30,45.99957),new Jrb.eq(31,47.00762),new Jrb.eq(32,48.01299),new Jrb.eq(33,49.02201)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(11,28.02851),new Jrb.eq(12,29.01411),new Jrb.eq(13,30.00477),new Jrb.eq(14,30.992416014),new Jrb.eq(15,31.985688908),new Jrb.eq(16,32.977451798),new Jrb.eq(17,33.973761967),new Jrb.eq(18,34.968852707),new Jrb.eq(19,35.968306945),new Jrb.eq(20,36.9659026),new Jrb.eq(21,37.96801055),new Jrb.eq(22,38.968007677),new Jrb.eq(23,39.970415555),new Jrb.eq(24,40.970650212),new Jrb.eq(25,41.973174994),new Jrb.eq(26,42.974203385),new Jrb.eq(27,43.978538712),new Jrb.eq(28,44.9797),new Jrb.eq(29,45.98412),new Jrb.eq(30,46.98795),new Jrb.eq(31,47.99485),new Jrb.eq(32,48.99989),new Jrb.eq(33,50.00773),new Jrb.eq(34,51.01353)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(12,30.02156),new Jrb.eq(13,31.012126),new Jrb.eq(14,31.99766066),new Jrb.eq(15,32.989928719),new Jrb.eq(16,33.980270118),new Jrb.eq(17,34.975256726),new Jrb.eq(18,35.967546282),new Jrb.eq(19,36.966775912),new Jrb.eq(20,37.962732161),new Jrb.eq(21,38.964313413),new Jrb.eq(22,39.962383123),new Jrb.eq(23,40.964500828),new Jrb.eq(24,41.963046386),new Jrb.eq(25,42.965670701),new Jrb.eq(26,43.965365269),new Jrb.eq(27,44.968094979),new Jrb.eq(28,45.968093467),new Jrb.eq(29,46.972186238),new Jrb.eq(30,47.97507),new Jrb.eq(31,48.98218),new Jrb.eq(32,49.98594),new Jrb.eq(33,50.99324),new Jrb.eq(34,51.99817),new Jrb.eq(35,53.006227)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(13,32.02192),new Jrb.eq(14,33.00726),new Jrb.eq(15,33.99841),new Jrb.eq(16,34.988011615),new Jrb.eq(17,35.981293405),new Jrb.eq(18,36.973376915),new Jrb.eq(19,37.969080107),new Jrb.eq(20,38.963706861),new Jrb.eq(21,39.963998672),new Jrb.eq(22,40.961825972),new Jrb.eq(23,41.962403059),new Jrb.eq(24,42.960715746),new Jrb.eq(25,43.961556146),new Jrb.eq(26,44.960699658),new Jrb.eq(27,45.961976203),new Jrb.eq(28,46.961677807),new Jrb.eq(29,47.965512946),new Jrb.eq(30,48.967450084),new Jrb.eq(31,49.972782832),new Jrb.eq(32,50.97638),new Jrb.eq(33,51.98261),new Jrb.eq(34,52.98712),new Jrb.eq(35,53.99399),new Jrb.eq(36,54.999388)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(14,34.01412),new Jrb.eq(15,35.004765),new Jrb.eq(16,35.993087234),new Jrb.eq(17,36.985871505),new Jrb.eq(18,37.976318637),new Jrb.eq(19,38.970717729),new Jrb.eq(20,39.962591155),new Jrb.eq(21,40.962278349),new Jrb.eq(22,41.958618337),new Jrb.eq(23,42.958766833),new Jrb.eq(24,43.955481094),new Jrb.eq(25,44.956185938),new Jrb.eq(26,45.953692759),new Jrb.eq(27,46.954546459),new Jrb.eq(28,47.952533512),new Jrb.eq(29,48.955673302),new Jrb.eq(30,49.957518286),new Jrb.eq(31,50.961474238),new Jrb.eq(32,51.9651),new Jrb.eq(33,52.97005),new Jrb.eq(34,53.97468),new Jrb.eq(35,54.98055),new Jrb.eq(36,55.98579),new Jrb.eq(37,56.992356)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(15,36.01492),new Jrb.eq(16,37.00305),new Jrb.eq(17,37.9947),new Jrb.eq(18,38.984790009),new Jrb.eq(19,39.977964014),new Jrb.eq(20,40.969251316),new Jrb.eq(21,41.965516761),new Jrb.eq(22,42.96115098),new Jrb.eq(23,43.959403048),new Jrb.eq(24,44.955910243),new Jrb.eq(25,45.95517025),new Jrb.eq(26,46.952408027),new Jrb.eq(27,47.952234991),new Jrb.eq(28,48.950024065),new Jrb.eq(29,49.952187008),new Jrb.eq(30,50.9536027),new Jrb.eq(31,51.95665),new Jrb.eq(32,52.95817),new Jrb.eq(33,53.963),new Jrb.eq(34,54.9694),new Jrb.eq(35,55.97266),new Jrb.eq(36,56.97704),new Jrb.eq(37,57.98307),new Jrb.eq(38,58.988041)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(16,38.00977),new Jrb.eq(17,39.001323),new Jrb.eq(18,39.990498907),new Jrb.eq(19,40.983131),new Jrb.eq(20,41.973031622),new Jrb.eq(21,42.968523342),new Jrb.eq(22,43.959690235),new Jrb.eq(23,44.958124349),new Jrb.eq(24,45.952629491),new Jrb.eq(25,46.951763792),new Jrb.eq(26,47.947947053),new Jrb.eq(27,48.947870789),new Jrb.eq(28,49.944792069),new Jrb.eq(29,50.946616017),new Jrb.eq(30,51.946898175),new Jrb.eq(31,52.949731709),new Jrb.eq(32,53.95087),new Jrb.eq(33,54.95512),new Jrb.eq(34,55.95799),new Jrb.eq(35,56.9643),new Jrb.eq(36,57.96611),new Jrb.eq(37,58.97196),new Jrb.eq(38,59.97564),new Jrb.eq(39,60.982018)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(17,40.01109),new Jrb.eq(18,40.99974),new Jrb.eq(19,41.99123),new Jrb.eq(20,42.98065),new Jrb.eq(21,43.9744),new Jrb.eq(22,44.965782286),new Jrb.eq(23,45.960199491),new Jrb.eq(24,46.954906918),new Jrb.eq(25,47.95225448),new Jrb.eq(26,48.948516914),new Jrb.eq(27,49.947162792),new Jrb.eq(28,50.943963675),new Jrb.eq(29,51.944779658),new Jrb.eq(30,52.944342517),new Jrb.eq(31,53.946444381),new Jrb.eq(32,54.947238194),new Jrb.eq(33,55.95036),new Jrb.eq(34,56.95236),new Jrb.eq(35,57.95665),new Jrb.eq(36,58.9593),new Jrb.eq(37,59.9645),new Jrb.eq(38,60.96741),new Jrb.eq(39,61.97314),new Jrb.eq(40,62.97675)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(18,42.00643),new Jrb.eq(19,42.997707),new Jrb.eq(20,43.98547),new Jrb.eq(21,44.97916),new Jrb.eq(22,45.968361649),new Jrb.eq(23,46.962906512),new Jrb.eq(24,47.954035861),new Jrb.eq(25,48.951341135),new Jrb.eq(26,49.946049607),new Jrb.eq(27,50.944771767),new Jrb.eq(28,51.940511904),new Jrb.eq(29,52.940653781),new Jrb.eq(30,53.938884921),new Jrb.eq(31,54.940844164),new Jrb.eq(32,55.940645238),new Jrb.eq(33,56.9437538),new Jrb.eq(34,57.94425),new Jrb.eq(35,58.94863),new Jrb.eq(36,59.94973),new Jrb.eq(37,60.95409),new Jrb.eq(38,61.9558),new Jrb.eq(39,62.96186),new Jrb.eq(40,63.9642),new Jrb.eq(41,64.97037)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(19,44.00687),new Jrb.eq(20,44.99451),new Jrb.eq(21,45.98672),new Jrb.eq(22,46.9761),new Jrb.eq(23,47.96887),new Jrb.eq(24,48.959623415),new Jrb.eq(25,49.95424396),new Jrb.eq(26,50.948215487),new Jrb.eq(27,51.945570079),new Jrb.eq(28,52.941294702),new Jrb.eq(29,53.940363247),new Jrb.eq(30,54.938049636),new Jrb.eq(31,55.938909366),new Jrb.eq(32,56.938287458),new Jrb.eq(33,57.939986451),new Jrb.eq(34,58.940447166),new Jrb.eq(35,59.943193998),new Jrb.eq(36,60.94446),new Jrb.eq(37,61.94797),new Jrb.eq(38,62.94981),new Jrb.eq(39,63.95373),new Jrb.eq(40,64.9561),new Jrb.eq(41,65.96082),new Jrb.eq(42,66.96382)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(19,45.01456),new Jrb.eq(20,46.00081),new Jrb.eq(21,46.99289),new Jrb.eq(22,47.98056),new Jrb.eq(23,48.97361),new Jrb.eq(24,49.962993316),new Jrb.eq(25,50.956824936),new Jrb.eq(26,51.948116526),new Jrb.eq(27,52.945312282),new Jrb.eq(28,53.939614836),new Jrb.eq(29,54.938298029),new Jrb.eq(30,55.934942133),new Jrb.eq(31,56.935398707),new Jrb.eq(32,57.933280458),new Jrb.eq(33,58.934880493),new Jrb.eq(34,59.934076943),new Jrb.eq(35,60.936749461),new Jrb.eq(36,61.936770495),new Jrb.eq(37,62.940118442),new Jrb.eq(38,63.94087),new Jrb.eq(39,64.94494),new Jrb.eq(40,65.94598),new Jrb.eq(41,66.95),new Jrb.eq(42,67.95251),new Jrb.eq(43,68.9577)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(21,48.00176),new Jrb.eq(22,48.98972),new Jrb.eq(23,49.98154),new Jrb.eq(24,50.97072),new Jrb.eq(25,51.96359),new Jrb.eq(26,52.954224985),new Jrb.eq(27,53.948464147),new Jrb.eq(28,54.942003149),new Jrb.eq(29,55.939843937),new Jrb.eq(30,56.936296235),new Jrb.eq(31,57.935757571),new Jrb.eq(32,58.933200194),new Jrb.eq(33,59.933822196),new Jrb.eq(34,60.932479381),new Jrb.eq(35,61.934054212),new Jrb.eq(36,62.933615218),new Jrb.eq(37,63.935813523),new Jrb.eq(38,64.936484581),new Jrb.eq(39,65.939825412),new Jrb.eq(40,66.94061),new Jrb.eq(41,67.94436),new Jrb.eq(42,68.9452),new Jrb.eq(43,69.94981),new Jrb.eq(44,70.95173),new Jrb.eq(45,71.95641)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(22,49.99593),new Jrb.eq(23,50.98772),new Jrb.eq(24,51.97568),new Jrb.eq(25,52.96846),new Jrb.eq(26,53.957910508),new Jrb.eq(27,54.951336329),new Jrb.eq(28,55.942136339),new Jrb.eq(29,56.939800489),new Jrb.eq(30,57.935347922),new Jrb.eq(31,58.934351553),new Jrb.eq(32,59.930790633),new Jrb.eq(33,60.931060442),new Jrb.eq(34,61.928348763),new Jrb.eq(35,62.929672948),new Jrb.eq(36,63.927969574),new Jrb.eq(37,64.930088013),new Jrb.eq(38,65.929115232),new Jrb.eq(39,66.931569638),new Jrb.eq(40,67.931844932),new Jrb.eq(41,68.935181837),new Jrb.eq(42,69.93614),new Jrb.eq(43,70.94),new Jrb.eq(44,71.9413),new Jrb.eq(45,72.94608),new Jrb.eq(46,73.94791),new Jrb.eq(47,74.95297),new Jrb.eq(48,75.95533),new Jrb.eq(49,76.96083),new Jrb.eq(50,77.9638)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(23,51.99718),new Jrb.eq(24,52.98555),new Jrb.eq(25,53.97671),new Jrb.eq(26,54.96605),new Jrb.eq(27,55.95856),new Jrb.eq(28,56.949215695),new Jrb.eq(29,57.944540734),new Jrb.eq(30,58.939504114),new Jrb.eq(31,59.937368123),new Jrb.eq(32,60.933462181),new Jrb.eq(33,61.932587299),new Jrb.eq(34,62.929601079),new Jrb.eq(35,63.929767865),new Jrb.eq(36,64.927793707),new Jrb.eq(37,65.928873041),new Jrb.eq(38,66.927750294),new Jrb.eq(39,67.929637875),new Jrb.eq(40,68.929425281),new Jrb.eq(41,69.932409287),new Jrb.eq(42,70.932619818),new Jrb.eq(43,71.93552),new Jrb.eq(44,72.93649),new Jrb.eq(45,73.9402),new Jrb.eq(46,74.9417),new Jrb.eq(47,75.94599),new Jrb.eq(48,76.94795),new Jrb.eq(49,77.95281),new Jrb.eq(50,78.95528),new Jrb.eq(51,79.96189)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(24,53.99295),new Jrb.eq(25,54.98398),new Jrb.eq(26,55.97238),new Jrb.eq(27,56.96491),new Jrb.eq(28,57.954596465),new Jrb.eq(29,58.949267074),new Jrb.eq(30,59.941832031),new Jrb.eq(31,60.939513907),new Jrb.eq(32,61.934334132),new Jrb.eq(33,62.933215563),new Jrb.eq(34,63.929146578),new Jrb.eq(35,64.929245079),new Jrb.eq(36,65.926036763),new Jrb.eq(37,66.927130859),new Jrb.eq(38,67.924847566),new Jrb.eq(39,68.926553538),new Jrb.eq(40,69.92532487),new Jrb.eq(41,70.927727195),new Jrb.eq(42,71.926861122),new Jrb.eq(43,72.929779469),new Jrb.eq(44,73.929458261),new Jrb.eq(45,74.932937379),new Jrb.eq(46,75.933394207),new Jrb.eq(47,76.937085857),new Jrb.eq(48,77.938569576),new Jrb.eq(49,78.942095175),new Jrb.eq(50,79.944414722),new Jrb.eq(51,80.95048),new Jrb.eq(52,81.95484)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(25,55.99491),new Jrb.eq(26,56.98293),new Jrb.eq(27,57.97425),new Jrb.eq(28,58.96337),new Jrb.eq(29,59.95706),new Jrb.eq(30,60.94917),new Jrb.eq(31,61.944179608),new Jrb.eq(32,62.939141527),new Jrb.eq(33,63.936838307),new Jrb.eq(34,64.932739322),new Jrb.eq(35,65.931592355),new Jrb.eq(36,66.928204915),new Jrb.eq(37,67.927983497),new Jrb.eq(38,68.925580912),new Jrb.eq(39,69.926027741),new Jrb.eq(40,70.92470501),new Jrb.eq(41,71.92636935),new Jrb.eq(42,72.925169832),new Jrb.eq(43,73.926940999),new Jrb.eq(44,74.926500645),new Jrb.eq(45,75.928928262),new Jrb.eq(46,76.929281189),new Jrb.eq(47,77.93165595),new Jrb.eq(48,78.932916371),new Jrb.eq(49,79.936588154),new Jrb.eq(50,80.937752955),new Jrb.eq(51,81.94316),new Jrb.eq(52,82.94687),new Jrb.eq(53,83.95234)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(26,57.99101),new Jrb.eq(27,58.98175),new Jrb.eq(28,59.97019),new Jrb.eq(29,60.96379),new Jrb.eq(30,61.95465),new Jrb.eq(31,62.94964),new Jrb.eq(32,63.941572638),new Jrb.eq(33,64.939440762),new Jrb.eq(34,65.933846798),new Jrb.eq(35,66.932738415),new Jrb.eq(36,67.928097266),new Jrb.eq(37,68.927972002),new Jrb.eq(38,69.924250365),new Jrb.eq(39,70.924953991),new Jrb.eq(40,71.922076184),new Jrb.eq(41,72.923459361),new Jrb.eq(42,73.921178213),new Jrb.eq(43,74.922859494),new Jrb.eq(44,75.921402716),new Jrb.eq(45,76.923548462),new Jrb.eq(46,77.922852886),new Jrb.eq(47,78.92540156),new Jrb.eq(48,79.925444764),new Jrb.eq(49,80.928821065),new Jrb.eq(50,81.929550326),new Jrb.eq(51,82.93451),new Jrb.eq(52,83.93731),new Jrb.eq(53,84.94269),new Jrb.eq(54,85.94627)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(27,59.99313),new Jrb.eq(28,60.98062),new Jrb.eq(29,61.9732),new Jrb.eq(30,62.96369),new Jrb.eq(31,63.957572),new Jrb.eq(32,64.949484),new Jrb.eq(33,65.944099147),new Jrb.eq(34,66.939190417),new Jrb.eq(35,67.936792976),new Jrb.eq(36,68.932280154),new Jrb.eq(37,69.930927811),new Jrb.eq(38,70.927114724),new Jrb.eq(39,71.926752647),new Jrb.eq(40,72.923825288),new Jrb.eq(41,73.923929076),new Jrb.eq(42,74.921596417),new Jrb.eq(43,75.922393933),new Jrb.eq(44,76.920647703),new Jrb.eq(45,77.921828577),new Jrb.eq(46,78.920948498),new Jrb.eq(47,79.922578162),new Jrb.eq(48,80.922132884),new Jrb.eq(49,81.924504668),new Jrb.eq(50,82.924980625),new Jrb.eq(51,83.92906),new Jrb.eq(52,84.93181),new Jrb.eq(53,85.93623),new Jrb.eq(54,86.93958),new Jrb.eq(55,87.94456),new Jrb.eq(56,88.94923)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(31,64.96466),new Jrb.eq(32,65.95521),new Jrb.eq(33,66.95009),new Jrb.eq(34,67.94187),new Jrb.eq(35,68.939562155),new Jrb.eq(36,69.933504),new Jrb.eq(37,70.931868378),new Jrb.eq(38,71.927112313),new Jrb.eq(39,72.9267668),new Jrb.eq(40,73.922476561),new Jrb.eq(41,74.922523571),new Jrb.eq(42,75.919214107),new Jrb.eq(43,76.91991461),new Jrb.eq(44,77.917309522),new Jrb.eq(45,78.918499802),new Jrb.eq(46,79.916521828),new Jrb.eq(47,80.917992931),new Jrb.eq(48,81.9167),new Jrb.eq(49,82.919119072),new Jrb.eq(50,83.918464523),new Jrb.eq(51,84.922244678),new Jrb.eq(52,85.924271165),new Jrb.eq(53,86.928520749),new Jrb.eq(54,87.931423982),new Jrb.eq(55,88.93602),new Jrb.eq(56,89.93942),new Jrb.eq(57,90.94537),new Jrb.eq(58,91.94933)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(32,66.96479),new Jrb.eq(33,67.958248),new Jrb.eq(34,68.950178),new Jrb.eq(35,69.944208),new Jrb.eq(36,70.939246),new Jrb.eq(37,71.936496876),new Jrb.eq(38,72.931794889),new Jrb.eq(39,73.929891152),new Jrb.eq(40,74.92577641),new Jrb.eq(41,75.924541974),new Jrb.eq(42,76.921380123),new Jrb.eq(43,77.92114613),new Jrb.eq(44,78.918337647),new Jrb.eq(45,79.918529952),new Jrb.eq(46,80.91629106),new Jrb.eq(47,81.916804666),new Jrb.eq(48,82.915180219),new Jrb.eq(49,83.916503685),new Jrb.eq(50,84.915608027),new Jrb.eq(51,85.918797162),new Jrb.eq(52,86.920710713),new Jrb.eq(53,87.924065908),new Jrb.eq(54,88.92638726),new Jrb.eq(55,89.930634988),new Jrb.eq(56,90.9339653),new Jrb.eq(57,91.939255258),new Jrb.eq(58,92.9431),new Jrb.eq(59,93.94868)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(33,68.96532),new Jrb.eq(34,69.95601),new Jrb.eq(35,70.95051),new Jrb.eq(36,71.94190754),new Jrb.eq(37,72.938931115),new Jrb.eq(38,73.933258225),new Jrb.eq(39,74.931033794),new Jrb.eq(40,75.925948304),new Jrb.eq(41,76.92466788),new Jrb.eq(42,77.920386271),new Jrb.eq(43,78.920082992),new Jrb.eq(44,79.91637804),new Jrb.eq(45,80.916592419),new Jrb.eq(46,81.913484601),new Jrb.eq(47,82.914135952),new Jrb.eq(48,83.911506627),new Jrb.eq(49,84.912526954),new Jrb.eq(50,85.910610313),new Jrb.eq(51,86.913354251),new Jrb.eq(52,87.914446951),new Jrb.eq(53,88.917632505),new Jrb.eq(54,89.919523803),new Jrb.eq(55,90.923442418),new Jrb.eq(56,91.926152752),new Jrb.eq(57,92.931265246),new Jrb.eq(58,93.934362),new Jrb.eq(59,94.93984),new Jrb.eq(60,95.94307),new Jrb.eq(61,96.94856)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(34,70.96532),new Jrb.eq(35,71.95908),new Jrb.eq(36,72.950366),new Jrb.eq(37,73.944470376),new Jrb.eq(38,74.938569199),new Jrb.eq(39,75.935071448),new Jrb.eq(40,76.930406599),new Jrb.eq(41,77.928141485),new Jrb.eq(42,78.923996719),new Jrb.eq(43,79.922519322),new Jrb.eq(44,80.918994165),new Jrb.eq(45,81.918207691),new Jrb.eq(46,82.915111951),new Jrb.eq(47,83.914384676),new Jrb.eq(48,84.911789341),new Jrb.eq(49,85.91116708),new Jrb.eq(50,86.909183465),new Jrb.eq(51,87.911318556),new Jrb.eq(52,88.912279939),new Jrb.eq(53,89.914808941),new Jrb.eq(54,90.91653416),new Jrb.eq(55,91.919725442),new Jrb.eq(56,92.922032765),new Jrb.eq(57,93.926407326),new Jrb.eq(58,94.92931926),new Jrb.eq(59,95.934283962),new Jrb.eq(60,96.937342863),new Jrb.eq(61,97.941703557),new Jrb.eq(62,98.945420616),new Jrb.eq(63,99.94987),new Jrb.eq(64,100.953195994),new Jrb.eq(65,101.95921)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(35,72.96597),new Jrb.eq(36,73.95631),new Jrb.eq(37,74.94992),new Jrb.eq(38,75.94161),new Jrb.eq(39,76.937761511),new Jrb.eq(40,77.932179362),new Jrb.eq(41,78.929707076),new Jrb.eq(42,79.924524588),new Jrb.eq(43,80.923213095),new Jrb.eq(44,81.918401258),new Jrb.eq(45,82.917555029),new Jrb.eq(46,83.913424778),new Jrb.eq(47,84.912932689),new Jrb.eq(48,85.909262351),new Jrb.eq(49,86.908879316),new Jrb.eq(50,87.905614339),new Jrb.eq(51,88.907452906),new Jrb.eq(52,89.907737596),new Jrb.eq(53,90.910209845),new Jrb.eq(54,91.911029895),new Jrb.eq(55,92.91402241),new Jrb.eq(56,93.915359856),new Jrb.eq(57,94.919358213),new Jrb.eq(58,95.921680473),new Jrb.eq(59,96.926148757),new Jrb.eq(60,97.928471177),new Jrb.eq(61,98.933315038),new Jrb.eq(62,99.935351729),new Jrb.eq(63,100.940517434),new Jrb.eq(64,101.943018795),new Jrb.eq(65,102.94895),new Jrb.eq(66,103.95233)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(38,76.94962),new Jrb.eq(39,77.9435),new Jrb.eq(40,78.937350712),new Jrb.eq(41,79.931982402),new Jrb.eq(42,80.929128719),new Jrb.eq(43,81.926792071),new Jrb.eq(44,82.922352572),new Jrb.eq(45,83.920387768),new Jrb.eq(46,84.916427076),new Jrb.eq(47,85.914887724),new Jrb.eq(48,86.910877833),new Jrb.eq(49,87.909503361),new Jrb.eq(50,88.905847902),new Jrb.eq(51,89.907151443),new Jrb.eq(52,90.907303415),new Jrb.eq(53,91.908946832),new Jrb.eq(54,92.909581582),new Jrb.eq(55,93.911594008),new Jrb.eq(56,94.912823709),new Jrb.eq(57,95.915897787),new Jrb.eq(58,96.918131017),new Jrb.eq(59,97.922219525),new Jrb.eq(60,98.924634736),new Jrb.eq(61,99.927756402),new Jrb.eq(62,100.930313395),new Jrb.eq(63,101.933555501),new Jrb.eq(64,102.93694),new Jrb.eq(65,103.94145),new Jrb.eq(66,104.94509),new Jrb.eq(67,105.95022)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(39,78.94916),new Jrb.eq(40,79.94055),new Jrb.eq(41,80.936815296),new Jrb.eq(42,81.931086249),new Jrb.eq(43,82.92865213),new Jrb.eq(44,83.92325),new Jrb.eq(45,84.92146522),new Jrb.eq(46,85.916472851),new Jrb.eq(47,86.914816578),new Jrb.eq(48,87.910226179),new Jrb.eq(49,88.908888916),new Jrb.eq(50,89.904703679),new Jrb.eq(51,90.905644968),new Jrb.eq(52,91.905040106),new Jrb.eq(53,92.906475627),new Jrb.eq(54,93.906315765),new Jrb.eq(55,94.908042739),new Jrb.eq(56,95.908275675),new Jrb.eq(57,96.910950716),new Jrb.eq(58,97.912746366),new Jrb.eq(59,98.916511084),new Jrb.eq(60,99.917761704),new Jrb.eq(61,100.921139958),new Jrb.eq(62,101.922981089),new Jrb.eq(63,102.926597062),new Jrb.eq(64,103.92878),new Jrb.eq(65,104.93305),new Jrb.eq(66,105.93591),new Jrb.eq(67,106.94086),new Jrb.eq(68,107.94428)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(40,80.94905),new Jrb.eq(41,81.94313),new Jrb.eq(42,82.936703713),new Jrb.eq(43,83.93357),new Jrb.eq(44,84.927906486),new Jrb.eq(45,85.925037588),new Jrb.eq(46,86.920361435),new Jrb.eq(47,87.91833144),new Jrb.eq(48,88.913495503),new Jrb.eq(49,89.911264109),new Jrb.eq(50,90.906990538),new Jrb.eq(51,91.907193214),new Jrb.eq(52,92.906377543),new Jrb.eq(53,93.907283457),new Jrb.eq(54,94.906835178),new Jrb.eq(55,95.908100076),new Jrb.eq(56,96.908097144),new Jrb.eq(57,97.91033069),new Jrb.eq(58,98.911617864),new Jrb.eq(59,99.914181434),new Jrb.eq(60,100.915251567),new Jrb.eq(61,101.918037417),new Jrb.eq(62,102.919141297),new Jrb.eq(63,103.922459464),new Jrb.eq(64,104.923934023),new Jrb.eq(65,105.92819),new Jrb.eq(66,106.93031),new Jrb.eq(67,107.93501),new Jrb.eq(68,108.93763),new Jrb.eq(69,109.94268)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(41,82.94874),new Jrb.eq(42,83.94009),new Jrb.eq(43,84.93659),new Jrb.eq(44,85.930695167),new Jrb.eq(45,86.92732683),new Jrb.eq(46,87.921952728),new Jrb.eq(47,88.919480562),new Jrb.eq(48,89.913936161),new Jrb.eq(49,90.911750754),new Jrb.eq(50,91.90681048),new Jrb.eq(51,92.906812213),new Jrb.eq(52,93.905087578),new Jrb.eq(53,94.905841487),new Jrb.eq(54,95.904678904),new Jrb.eq(55,96.906021033),new Jrb.eq(56,97.905407846),new Jrb.eq(57,98.907711598),new Jrb.eq(58,99.907477149),new Jrb.eq(59,100.910346543),new Jrb.eq(60,101.910297162),new Jrb.eq(61,102.913204596),new Jrb.eq(62,103.913758387),new Jrb.eq(63,104.916972087),new Jrb.eq(64,105.918134284),new Jrb.eq(65,106.921694724),new Jrb.eq(66,107.923973837),new Jrb.eq(67,108.92781),new Jrb.eq(68,109.92973),new Jrb.eq(69,110.93451),new Jrb.eq(70,111.93684),new Jrb.eq(71,112.94203)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(42,84.94894),new Jrb.eq(43,85.94288),new Jrb.eq(44,86.93653),new Jrb.eq(45,87.93283),new Jrb.eq(46,88.92754288),new Jrb.eq(47,89.92355583),new Jrb.eq(48,90.9184282),new Jrb.eq(49,91.915259655),new Jrb.eq(50,92.910248473),new Jrb.eq(51,93.909656309),new Jrb.eq(52,94.907656454),new Jrb.eq(53,95.907870803),new Jrb.eq(54,96.906364843),new Jrb.eq(55,97.907215692),new Jrb.eq(56,98.906254554),new Jrb.eq(57,99.907657594),new Jrb.eq(58,100.90731438),new Jrb.eq(59,101.909212938),new Jrb.eq(60,102.909178805),new Jrb.eq(61,103.911444898),new Jrb.eq(62,104.911658043),new Jrb.eq(63,105.914355408),new Jrb.eq(64,106.915081691),new Jrb.eq(65,107.918479973),new Jrb.eq(66,108.919980998),new Jrb.eq(67,109.92339),new Jrb.eq(68,110.92505),new Jrb.eq(69,111.92924),new Jrb.eq(70,112.93133),new Jrb.eq(71,113.93588),new Jrb.eq(72,114.93828)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(43,86.94918),new Jrb.eq(44,87.94042),new Jrb.eq(45,88.93611),new Jrb.eq(46,89.92978),new Jrb.eq(47,90.926377434),new Jrb.eq(48,91.92012),new Jrb.eq(49,92.917051523),new Jrb.eq(50,93.911359569),new Jrb.eq(51,94.910412729),new Jrb.eq(52,95.907597681),new Jrb.eq(53,96.907554546),new Jrb.eq(54,97.905287111),new Jrb.eq(55,98.905939307),new Jrb.eq(56,99.904219664),new Jrb.eq(57,100.905582219),new Jrb.eq(58,101.904349503),new Jrb.eq(59,102.906323677),new Jrb.eq(60,103.905430145),new Jrb.eq(61,104.907750341),new Jrb.eq(62,105.907326913),new Jrb.eq(63,106.909907207),new Jrb.eq(64,107.910192211),new Jrb.eq(65,108.913201565),new Jrb.eq(66,109.913966185),new Jrb.eq(67,110.91756),new Jrb.eq(68,111.918821673),new Jrb.eq(69,112.92254),new Jrb.eq(70,113.923891981),new Jrb.eq(71,114.92831),new Jrb.eq(72,115.93016),new Jrb.eq(73,116.93479),new Jrb.eq(74,117.93703)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(44,88.94938),new Jrb.eq(45,89.94287),new Jrb.eq(46,90.93655),new Jrb.eq(47,91.93198),new Jrb.eq(48,92.92574),new Jrb.eq(49,93.921698),new Jrb.eq(50,94.915898541),new Jrb.eq(51,95.914518212),new Jrb.eq(52,96.911336643),new Jrb.eq(53,97.910716431),new Jrb.eq(54,98.908132101),new Jrb.eq(55,99.90811663),new Jrb.eq(56,100.906163526),new Jrb.eq(57,101.906842845),new Jrb.eq(58,102.905504182),new Jrb.eq(59,103.906655315),new Jrb.eq(60,104.905692444),new Jrb.eq(61,105.907284615),new Jrb.eq(62,106.90675054),new Jrb.eq(63,107.908730768),new Jrb.eq(64,108.908735621),new Jrb.eq(65,109.910949525),new Jrb.eq(66,110.91166),new Jrb.eq(67,111.913969253),new Jrb.eq(68,112.91542),new Jrb.eq(69,113.91734336),new Jrb.eq(70,114.920124676),new Jrb.eq(71,115.922746643),new Jrb.eq(72,116.92535),new Jrb.eq(73,117.92943),new Jrb.eq(74,118.93136),new Jrb.eq(75,119.93578),new Jrb.eq(76,120.93808)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(45,90.94948),new Jrb.eq(46,91.94042),new Jrb.eq(47,92.93591),new Jrb.eq(48,93.92877),new Jrb.eq(49,94.92469),new Jrb.eq(50,95.91822194),new Jrb.eq(51,96.916478921),new Jrb.eq(52,97.912720751),new Jrb.eq(53,98.911767757),new Jrb.eq(54,99.908504596),new Jrb.eq(55,100.908289144),new Jrb.eq(56,101.905607716),new Jrb.eq(57,102.906087204),new Jrb.eq(58,103.904034912),new Jrb.eq(59,104.905084046),new Jrb.eq(60,105.903483087),new Jrb.eq(61,106.905128453),new Jrb.eq(62,107.903894451),new Jrb.eq(63,108.905953535),new Jrb.eq(64,109.905152385),new Jrb.eq(65,110.907643952),new Jrb.eq(66,111.907313277),new Jrb.eq(67,112.910151346),new Jrb.eq(68,113.910365322),new Jrb.eq(69,114.91368341),new Jrb.eq(70,115.914158288),new Jrb.eq(71,116.91784),new Jrb.eq(72,117.918983915),new Jrb.eq(73,118.92268),new Jrb.eq(74,119.92403),new Jrb.eq(75,120.92818),new Jrb.eq(76,121.9298),new Jrb.eq(77,122.93426)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(47,93.94278),new Jrb.eq(48,94.93548),new Jrb.eq(49,95.93068),new Jrb.eq(50,96.924),new Jrb.eq(51,97.921759995),new Jrb.eq(52,98.917597103),new Jrb.eq(53,99.916069387),new Jrb.eq(54,100.912802135),new Jrb.eq(55,101.911999996),new Jrb.eq(56,102.908972453),new Jrb.eq(57,103.908628228),new Jrb.eq(58,104.906528234),new Jrb.eq(59,105.906666431),new Jrb.eq(60,106.90509302),new Jrb.eq(61,107.905953705),new Jrb.eq(62,108.904755514),new Jrb.eq(63,109.90611046),new Jrb.eq(64,110.905294679),new Jrb.eq(65,111.907004132),new Jrb.eq(66,112.906565708),new Jrb.eq(67,113.908807907),new Jrb.eq(68,114.908762282),new Jrb.eq(69,115.911359558),new Jrb.eq(70,116.911684187),new Jrb.eq(71,117.914582383),new Jrb.eq(72,118.915666045),new Jrb.eq(73,119.918788609),new Jrb.eq(74,120.919851074),new Jrb.eq(75,121.92332),new Jrb.eq(76,122.9249),new Jrb.eq(77,123.92853),new Jrb.eq(78,124.93054),new Jrb.eq(79,125.9345),new Jrb.eq(80,126.93688)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(48,95.93977),new Jrb.eq(49,96.93494),new Jrb.eq(50,97.927579),new Jrb.eq(51,98.92501),new Jrb.eq(52,99.920230232),new Jrb.eq(53,100.918681442),new Jrb.eq(54,101.914777255),new Jrb.eq(55,102.913418952),new Jrb.eq(56,103.909848091),new Jrb.eq(57,104.909467818),new Jrb.eq(58,105.906458007),new Jrb.eq(59,106.906614232),new Jrb.eq(60,107.904183403),new Jrb.eq(61,108.904985569),new Jrb.eq(62,109.903005578),new Jrb.eq(63,110.904181628),new Jrb.eq(64,111.902757226),new Jrb.eq(65,112.904400947),new Jrb.eq(66,113.903358121),new Jrb.eq(67,114.905430553),new Jrb.eq(68,115.904755434),new Jrb.eq(69,116.907218242),new Jrb.eq(70,117.906914144),new Jrb.eq(71,118.909922582),new Jrb.eq(72,119.909851352),new Jrb.eq(73,120.91298039),new Jrb.eq(74,121.9135),new Jrb.eq(75,122.917003675),new Jrb.eq(76,123.917648302),new Jrb.eq(77,124.92124717),new Jrb.eq(78,125.922353996),new Jrb.eq(79,126.926434822),new Jrb.eq(80,127.927760617),new Jrb.eq(81,128.93226),new Jrb.eq(82,129.93398)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(49,97.94224),new Jrb.eq(50,98.93461),new Jrb.eq(51,99.931149033),new Jrb.eq(52,100.92656),new Jrb.eq(53,101.924707541),new Jrb.eq(54,102.919913896),new Jrb.eq(55,103.918338416),new Jrb.eq(56,104.914673434),new Jrb.eq(57,105.913461134),new Jrb.eq(58,106.910292195),new Jrb.eq(59,107.909719683),new Jrb.eq(60,108.907154078),new Jrb.eq(61,109.907168783),new Jrb.eq(62,110.905110677),new Jrb.eq(63,111.905533338),new Jrb.eq(64,112.904061223),new Jrb.eq(65,113.904916758),new Jrb.eq(66,114.903878328),new Jrb.eq(67,115.905259995),new Jrb.eq(68,116.904515731),new Jrb.eq(69,117.906354623),new Jrb.eq(70,118.905846334),new Jrb.eq(71,119.907961505),new Jrb.eq(72,120.907848847),new Jrb.eq(73,121.910277103),new Jrb.eq(74,122.910438951),new Jrb.eq(75,123.913175916),new Jrb.eq(76,124.913601387),new Jrb.eq(77,125.916464532),new Jrb.eq(78,126.917344048),new Jrb.eq(79,127.920170658),new Jrb.eq(80,128.921657958),new Jrb.eq(81,129.924854941),new Jrb.eq(82,130.926767408),new Jrb.eq(83,131.932919005),new Jrb.eq(84,132.93834),new Jrb.eq(85,133.94466)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(50,99.938954),new Jrb.eq(51,100.93606),new Jrb.eq(52,101.93049),new Jrb.eq(53,102.92813),new Jrb.eq(54,103.923185469),new Jrb.eq(55,104.921390409),new Jrb.eq(56,105.916880472),new Jrb.eq(57,106.915666702),new Jrb.eq(58,107.911965339),new Jrb.eq(59,108.911286879),new Jrb.eq(60,109.907852688),new Jrb.eq(61,110.907735404),new Jrb.eq(62,111.90482081),new Jrb.eq(63,112.905173373),new Jrb.eq(64,113.902781816),new Jrb.eq(65,114.903345973),new Jrb.eq(66,115.901744149),new Jrb.eq(67,116.902953765),new Jrb.eq(68,117.901606328),new Jrb.eq(69,118.90330888),new Jrb.eq(70,119.902196571),new Jrb.eq(71,120.904236867),new Jrb.eq(72,121.903440138),new Jrb.eq(73,122.905721901),new Jrb.eq(74,123.90527463),new Jrb.eq(75,124.907784924),new Jrb.eq(76,125.907653953),new Jrb.eq(77,126.91035098),new Jrb.eq(78,127.910534953),new Jrb.eq(79,128.913439976),new Jrb.eq(80,129.913852185),new Jrb.eq(81,130.916919144),new Jrb.eq(82,131.917744455),new Jrb.eq(83,132.923814085),new Jrb.eq(84,133.928463576),new Jrb.eq(85,134.93473),new Jrb.eq(86,135.93934),new Jrb.eq(87,136.94579)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(52,102.94012),new Jrb.eq(53,103.936287),new Jrb.eq(54,104.931528593),new Jrb.eq(55,105.928183134),new Jrb.eq(56,106.92415),new Jrb.eq(57,107.92216),new Jrb.eq(58,108.918136092),new Jrb.eq(59,109.917533911),new Jrb.eq(60,110.912534147),new Jrb.eq(61,111.91239464),new Jrb.eq(62,112.909377941),new Jrb.eq(63,113.909095876),new Jrb.eq(64,114.906598812),new Jrb.eq(65,115.906797235),new Jrb.eq(66,116.90483959),new Jrb.eq(67,117.905531885),new Jrb.eq(68,118.90394646),new Jrb.eq(69,119.905074315),new Jrb.eq(70,120.903818044),new Jrb.eq(71,121.905175415),new Jrb.eq(72,122.904215696),new Jrb.eq(73,123.905937525),new Jrb.eq(74,124.905247804),new Jrb.eq(75,125.907248153),new Jrb.eq(76,126.906914564),new Jrb.eq(77,127.90916733),new Jrb.eq(78,128.909150092),new Jrb.eq(79,129.911546459),new Jrb.eq(80,130.911946487),new Jrb.eq(81,131.914413247),new Jrb.eq(82,132.915236466),new Jrb.eq(83,133.920551554),new Jrb.eq(84,134.925167962),new Jrb.eq(85,135.93066),new Jrb.eq(86,136.93531),new Jrb.eq(87,137.94096),new Jrb.eq(88,138.94571)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(54,105.937702),new Jrb.eq(55,106.935036),new Jrb.eq(56,107.929486838),new Jrb.eq(57,108.927456483),new Jrb.eq(58,109.922407164),new Jrb.eq(59,110.921120589),new Jrb.eq(60,111.917061617),new Jrb.eq(61,112.915452551),new Jrb.eq(62,113.912498025),new Jrb.eq(63,114.911578627),new Jrb.eq(64,115.908420253),new Jrb.eq(65,116.90863418),new Jrb.eq(66,117.905825187),new Jrb.eq(67,118.90640811),new Jrb.eq(68,119.904019891),new Jrb.eq(69,120.904929815),new Jrb.eq(70,121.903047064),new Jrb.eq(71,122.904272951),new Jrb.eq(72,123.902819466),new Jrb.eq(73,124.904424718),new Jrb.eq(74,125.903305543),new Jrb.eq(75,126.90521729),new Jrb.eq(76,127.904461383),new Jrb.eq(77,128.906595593),new Jrb.eq(78,129.906222753),new Jrb.eq(79,130.90852188),new Jrb.eq(80,131.908523782),new Jrb.eq(81,132.910939068),new Jrb.eq(82,133.911540546),new Jrb.eq(83,134.916450782),new Jrb.eq(84,135.920103155),new Jrb.eq(85,136.925324769),new Jrb.eq(86,137.92922),new Jrb.eq(87,138.93473),new Jrb.eq(88,139.9387),new Jrb.eq(89,140.94439),new Jrb.eq(90,141.9485)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(55,107.943291),new Jrb.eq(56,108.938191658),new Jrb.eq(57,109.934634181),new Jrb.eq(58,110.930276),new Jrb.eq(59,111.92797),new Jrb.eq(60,112.923644245),new Jrb.eq(61,113.92185),new Jrb.eq(62,114.918272),new Jrb.eq(63,115.916735014),new Jrb.eq(64,116.913647692),new Jrb.eq(65,117.91337523),new Jrb.eq(66,118.910180837),new Jrb.eq(67,119.910047843),new Jrb.eq(68,120.907366063),new Jrb.eq(69,121.907592451),new Jrb.eq(70,122.905597944),new Jrb.eq(71,123.906211423),new Jrb.eq(72,124.90462415),new Jrb.eq(73,125.905619387),new Jrb.eq(74,126.90446842),new Jrb.eq(75,127.905805254),new Jrb.eq(76,128.904987487),new Jrb.eq(77,129.906674018),new Jrb.eq(78,130.906124168),new Jrb.eq(79,131.907994525),new Jrb.eq(80,132.907806465),new Jrb.eq(81,133.909876552),new Jrb.eq(82,134.91005031),new Jrb.eq(83,135.914655105),new Jrb.eq(84,136.917872653),new Jrb.eq(85,137.922383666),new Jrb.eq(86,138.926093402),new Jrb.eq(87,139.93121),new Jrb.eq(88,140.93483),new Jrb.eq(89,141.94018),new Jrb.eq(90,142.94407),new Jrb.eq(91,143.94961)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(56,109.944476),new Jrb.eq(57,110.941632),new Jrb.eq(58,111.93566535),new Jrb.eq(59,112.933382836),new Jrb.eq(60,113.928145),new Jrb.eq(61,114.926979032),new Jrb.eq(62,115.921394197),new Jrb.eq(63,116.920564355),new Jrb.eq(64,117.91657092),new Jrb.eq(65,118.915554295),new Jrb.eq(66,119.91215199),new Jrb.eq(67,120.911386497),new Jrb.eq(68,121.908548396),new Jrb.eq(69,122.908470748),new Jrb.eq(70,123.905895774),new Jrb.eq(71,124.906398236),new Jrb.eq(72,125.904268868),new Jrb.eq(73,126.905179581),new Jrb.eq(74,127.903530436),new Jrb.eq(75,128.904779458),new Jrb.eq(76,129.903507903),new Jrb.eq(77,130.90508192),new Jrb.eq(78,131.904154457),new Jrb.eq(79,132.90590566),new Jrb.eq(80,133.905394504),new Jrb.eq(81,134.907207499),new Jrb.eq(82,135.907219526),new Jrb.eq(83,136.911562939),new Jrb.eq(84,137.913988549),new Jrb.eq(85,138.918786859),new Jrb.eq(86,139.921635665),new Jrb.eq(87,140.926646282),new Jrb.eq(88,141.929702981),new Jrb.eq(89,142.93489),new Jrb.eq(90,143.93823),new Jrb.eq(91,144.94367),new Jrb.eq(92,145.9473),new Jrb.eq(93,146.95301)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(57,111.950331),new Jrb.eq(58,112.944535512),new Jrb.eq(59,113.940841319),new Jrb.eq(60,114.935939),new Jrb.eq(61,115.932914152),new Jrb.eq(62,116.928639484),new Jrb.eq(63,117.926554883),new Jrb.eq(64,118.922370879),new Jrb.eq(65,119.920678219),new Jrb.eq(66,120.917183637),new Jrb.eq(67,121.916121946),new Jrb.eq(68,122.912990168),new Jrb.eq(69,123.912245731),new Jrb.eq(70,124.909724871),new Jrb.eq(71,125.909447953),new Jrb.eq(72,126.9074176),new Jrb.eq(73,127.907747919),new Jrb.eq(74,128.906063369),new Jrb.eq(75,129.906706163),new Jrb.eq(76,130.905460232),new Jrb.eq(77,131.906429799),new Jrb.eq(78,132.90544687),new Jrb.eq(79,133.906713419),new Jrb.eq(80,134.905971903),new Jrb.eq(81,135.907305741),new Jrb.eq(82,136.907083505),new Jrb.eq(83,137.911010537),new Jrb.eq(84,138.913357921),new Jrb.eq(85,139.917277075),new Jrb.eq(86,140.920043984),new Jrb.eq(87,141.924292317),new Jrb.eq(88,142.927330292),new Jrb.eq(89,143.932027373),new Jrb.eq(90,144.935388226),new Jrb.eq(91,145.940162028),new Jrb.eq(92,146.943864435),new Jrb.eq(93,147.948899539),new Jrb.eq(94,148.95272),new Jrb.eq(95,149.95797),new Jrb.eq(96,150.962)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(58,113.950941),new Jrb.eq(59,114.94771),new Jrb.eq(60,115.94168),new Jrb.eq(61,116.937700229),new Jrb.eq(62,117.93344),new Jrb.eq(63,118.931051927),new Jrb.eq(64,119.926045941),new Jrb.eq(65,120.924485908),new Jrb.eq(66,121.92026),new Jrb.eq(67,122.91885),new Jrb.eq(68,123.915088437),new Jrb.eq(69,124.914620234),new Jrb.eq(70,125.911244146),new Jrb.eq(71,126.911121328),new Jrb.eq(72,127.90830887),new Jrb.eq(73,128.908673749),new Jrb.eq(74,129.906310478),new Jrb.eq(75,130.906930798),new Jrb.eq(76,131.905056152),new Jrb.eq(77,132.906002368),new Jrb.eq(78,133.904503347),new Jrb.eq(79,134.905682749),new Jrb.eq(80,135.904570109),new Jrb.eq(81,136.905821414),new Jrb.eq(82,137.905241273),new Jrb.eq(83,138.908835384),new Jrb.eq(84,139.910599485),new Jrb.eq(85,140.914406439),new Jrb.eq(86,141.916448175),new Jrb.eq(87,142.920617184),new Jrb.eq(88,143.922940468),new Jrb.eq(89,144.926923807),new Jrb.eq(90,145.930106645),new Jrb.eq(91,146.933992519),new Jrb.eq(92,147.937682377),new Jrb.eq(93,148.94246),new Jrb.eq(94,149.94562),new Jrb.eq(95,150.9507),new Jrb.eq(96,151.95416),new Jrb.eq(97,152.95961)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(60,116.95001),new Jrb.eq(61,117.94657),new Jrb.eq(62,118.94099),new Jrb.eq(63,119.93807),new Jrb.eq(64,120.93301),new Jrb.eq(65,121.93071),new Jrb.eq(66,122.92624),new Jrb.eq(67,123.92453),new Jrb.eq(68,124.92067),new Jrb.eq(69,125.91937),new Jrb.eq(70,126.91616),new Jrb.eq(71,127.91544794),new Jrb.eq(72,128.912667334),new Jrb.eq(73,129.91232),new Jrb.eq(74,130.910108489),new Jrb.eq(75,131.910110399),new Jrb.eq(76,132.908396372),new Jrb.eq(77,133.908489607),new Jrb.eq(78,134.906971003),new Jrb.eq(79,135.907651181),new Jrb.eq(80,136.906465656),new Jrb.eq(81,137.907106826),new Jrb.eq(82,138.90634816),new Jrb.eq(83,139.909472552),new Jrb.eq(84,140.910957016),new Jrb.eq(85,141.914074489),new Jrb.eq(86,142.916058646),new Jrb.eq(87,143.919591666),new Jrb.eq(88,144.92163837),new Jrb.eq(89,145.925700146),new Jrb.eq(90,146.927819639),new Jrb.eq(91,147.932191197),new Jrb.eq(92,148.93437),new Jrb.eq(93,149.93857),new Jrb.eq(94,150.94156),new Jrb.eq(95,151.94611),new Jrb.eq(96,152.94945),new Jrb.eq(97,153.9544),new Jrb.eq(98,154.95813)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(61,118.95276),new Jrb.eq(62,119.94664),new Jrb.eq(63,120.94367),new Jrb.eq(64,121.93801),new Jrb.eq(65,122.93551),new Jrb.eq(66,123.93052),new Jrb.eq(67,124.92854),new Jrb.eq(68,125.9241),new Jrb.eq(69,126.92275),new Jrb.eq(70,127.91887),new Jrb.eq(71,128.918679183),new Jrb.eq(72,129.914339361),new Jrb.eq(73,130.914424137),new Jrb.eq(74,131.91149),new Jrb.eq(75,132.91155),new Jrb.eq(76,133.909026379),new Jrb.eq(77,134.909145555),new Jrb.eq(78,135.907143574),new Jrb.eq(79,136.907777634),new Jrb.eq(80,137.905985574),new Jrb.eq(81,138.906646605),new Jrb.eq(82,139.905434035),new Jrb.eq(83,140.908271103),new Jrb.eq(84,141.909239733),new Jrb.eq(85,142.912381158),new Jrb.eq(86,143.913642686),new Jrb.eq(87,144.917227871),new Jrb.eq(88,145.918689722),new Jrb.eq(89,146.922510962),new Jrb.eq(90,147.924394738),new Jrb.eq(91,148.928289207),new Jrb.eq(92,149.930226399),new Jrb.eq(93,150.93404),new Jrb.eq(94,151.93638),new Jrb.eq(95,152.94058),new Jrb.eq(96,153.94332),new Jrb.eq(97,154.94804),new Jrb.eq(98,155.95126),new Jrb.eq(99,156.95634)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(62,120.955364),new Jrb.eq(63,121.95165),new Jrb.eq(64,122.94596),new Jrb.eq(65,123.94296),new Jrb.eq(66,124.93783),new Jrb.eq(67,125.93531),new Jrb.eq(68,126.93083),new Jrb.eq(69,127.9288),new Jrb.eq(70,128.92486),new Jrb.eq(71,129.92338),new Jrb.eq(72,130.920060245),new Jrb.eq(73,131.91912),new Jrb.eq(74,132.9162),new Jrb.eq(75,133.915672),new Jrb.eq(76,134.91313914),new Jrb.eq(77,135.912646935),new Jrb.eq(78,136.910678351),new Jrb.eq(79,137.910748891),new Jrb.eq(80,138.908932181),new Jrb.eq(81,139.909071204),new Jrb.eq(82,140.907647726),new Jrb.eq(83,141.910039865),new Jrb.eq(84,142.910812233),new Jrb.eq(85,143.913300595),new Jrb.eq(86,144.914506897),new Jrb.eq(87,145.917588016),new Jrb.eq(88,146.918979001),new Jrb.eq(89,147.922183237),new Jrb.eq(90,148.923791056),new Jrb.eq(91,149.926995031),new Jrb.eq(92,150.928227869),new Jrb.eq(93,151.9316),new Jrb.eq(94,152.93365),new Jrb.eq(95,153.93739),new Jrb.eq(96,154.93999),new Jrb.eq(97,155.94412),new Jrb.eq(98,156.94717),new Jrb.eq(99,157.95178),new Jrb.eq(100,158.95523)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(66,125.94307),new Jrb.eq(67,126.9405),new Jrb.eq(68,127.93539),new Jrb.eq(69,128.932385),new Jrb.eq(70,129.92878),new Jrb.eq(71,130.927102697),new Jrb.eq(72,131.92312),new Jrb.eq(73,132.92221),new Jrb.eq(74,133.918645),new Jrb.eq(75,134.91824),new Jrb.eq(76,135.915020542),new Jrb.eq(77,136.91463973),new Jrb.eq(78,137.91291745),new Jrb.eq(79,138.91192415),new Jrb.eq(80,139.909309824),new Jrb.eq(81,140.9096048),new Jrb.eq(82,141.907718643),new Jrb.eq(83,142.909809626),new Jrb.eq(84,143.910082629),new Jrb.eq(85,144.912568847),new Jrb.eq(86,145.913112139),new Jrb.eq(87,146.916095794),new Jrb.eq(88,147.916888516),new Jrb.eq(89,148.92014419),new Jrb.eq(90,149.920886563),new Jrb.eq(91,150.923824739),new Jrb.eq(92,151.924682428),new Jrb.eq(93,152.927694534),new Jrb.eq(94,153.929483295),new Jrb.eq(95,154.932629551),new Jrb.eq(96,155.9352),new Jrb.eq(97,156.93927),new Jrb.eq(98,157.94187),new Jrb.eq(99,158.94639),new Jrb.eq(100,159.94939),new Jrb.eq(101,160.95433)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(67,127.94826),new Jrb.eq(68,128.94316),new Jrb.eq(69,129.94045),new Jrb.eq(70,130.9358),new Jrb.eq(71,131.93375),new Jrb.eq(72,132.92972),new Jrb.eq(73,133.92849),new Jrb.eq(74,134.924617),new Jrb.eq(75,135.923447865),new Jrb.eq(76,136.920713),new Jrb.eq(77,137.920432261),new Jrb.eq(78,138.916759814),new Jrb.eq(79,139.915801649),new Jrb.eq(80,140.913606636),new Jrb.eq(81,141.912950738),new Jrb.eq(82,142.910927571),new Jrb.eq(83,143.912585768),new Jrb.eq(84,144.912743879),new Jrb.eq(85,145.914692165),new Jrb.eq(86,146.915133898),new Jrb.eq(87,147.917467786),new Jrb.eq(88,148.918329195),new Jrb.eq(89,149.920979477),new Jrb.eq(90,150.921202693),new Jrb.eq(91,151.923490557),new Jrb.eq(92,152.924113189),new Jrb.eq(93,153.926547019),new Jrb.eq(94,154.928097047),new Jrb.eq(95,155.931060357),new Jrb.eq(96,156.9332),new Jrb.eq(97,157.93669),new Jrb.eq(98,158.93913),new Jrb.eq(99,159.94299),new Jrb.eq(100,160.94586),new Jrb.eq(101,161.95029),new Jrb.eq(102,162.95352)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(68,129.94863),new Jrb.eq(69,130.94589),new Jrb.eq(70,131.94082),new Jrb.eq(71,132.93873),new Jrb.eq(72,133.93402),new Jrb.eq(73,134.93235),new Jrb.eq(74,135.9283),new Jrb.eq(75,136.927046709),new Jrb.eq(76,137.92354),new Jrb.eq(77,138.922302),new Jrb.eq(78,139.918991),new Jrb.eq(79,140.918468512),new Jrb.eq(80,141.915193274),new Jrb.eq(81,142.914623555),new Jrb.eq(82,143.91199473),new Jrb.eq(83,144.913405611),new Jrb.eq(84,145.91303676),new Jrb.eq(85,146.914893275),new Jrb.eq(86,147.914817914),new Jrb.eq(87,148.917179521),new Jrb.eq(88,149.917271454),new Jrb.eq(89,150.919928351),new Jrb.eq(90,151.919728244),new Jrb.eq(91,152.922093907),new Jrb.eq(92,153.922205303),new Jrb.eq(93,154.92463594),new Jrb.eq(94,155.925526236),new Jrb.eq(95,156.928354506),new Jrb.eq(96,157.929987938),new Jrb.eq(97,158.9332),new Jrb.eq(98,159.93514),new Jrb.eq(99,160.93883),new Jrb.eq(100,161.94122),new Jrb.eq(101,162.94536),new Jrb.eq(102,163.94828),new Jrb.eq(103,164.95298)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(69,131.95416),new Jrb.eq(70,132.9489),new Jrb.eq(71,133.94632),new Jrb.eq(72,134.94172),new Jrb.eq(73,135.9395),new Jrb.eq(74,136.93521),new Jrb.eq(75,137.93345),new Jrb.eq(76,138.92882915),new Jrb.eq(77,139.928083921),new Jrb.eq(78,140.924885867),new Jrb.eq(79,141.923400033),new Jrb.eq(80,142.920286634),new Jrb.eq(81,143.918774116),new Jrb.eq(82,144.916261285),new Jrb.eq(83,145.917199714),new Jrb.eq(84,146.916741206),new Jrb.eq(85,147.918153775),new Jrb.eq(86,148.917925922),new Jrb.eq(87,149.919698294),new Jrb.eq(88,150.919846022),new Jrb.eq(89,151.921740399),new Jrb.eq(90,152.921226219),new Jrb.eq(91,153.922975386),new Jrb.eq(92,154.922889429),new Jrb.eq(93,155.924750855),new Jrb.eq(94,156.925419435),new Jrb.eq(95,157.927841923),new Jrb.eq(96,158.9290845),new Jrb.eq(97,159.931460406),new Jrb.eq(98,160.93368),new Jrb.eq(99,161.93704),new Jrb.eq(100,162.93921),new Jrb.eq(101,163.94299),new Jrb.eq(102,164.94572),new Jrb.eq(103,165.94997),new Jrb.eq(104,166.95305)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(72,135.94707),new Jrb.eq(73,136.94465),new Jrb.eq(74,137.93997),new Jrb.eq(75,138.93808),new Jrb.eq(76,139.933236934),new Jrb.eq(77,140.93221),new Jrb.eq(78,141.927908919),new Jrb.eq(79,142.926738636),new Jrb.eq(80,143.923390357),new Jrb.eq(81,144.921687498),new Jrb.eq(82,145.918305344),new Jrb.eq(83,146.919089446),new Jrb.eq(84,147.918109771),new Jrb.eq(85,148.919336427),new Jrb.eq(86,149.918655455),new Jrb.eq(87,150.920344273),new Jrb.eq(88,151.919787882),new Jrb.eq(89,152.921746283),new Jrb.eq(90,153.920862271),new Jrb.eq(91,154.922618801),new Jrb.eq(92,155.922119552),new Jrb.eq(93,156.923956686),new Jrb.eq(94,157.924100533),new Jrb.eq(95,158.926385075),new Jrb.eq(96,159.927050616),new Jrb.eq(97,160.929665688),new Jrb.eq(98,161.930981211),new Jrb.eq(99,162.93399),new Jrb.eq(100,163.93586),new Jrb.eq(101,164.93938),new Jrb.eq(102,165.9416),new Jrb.eq(103,166.94557),new Jrb.eq(104,167.94836),new Jrb.eq(105,168.95287)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(73,137.95287),new Jrb.eq(74,138.94803),new Jrb.eq(75,139.945367985),new Jrb.eq(76,140.94116),new Jrb.eq(77,141.939073781),new Jrb.eq(78,142.93475),new Jrb.eq(79,143.93253),new Jrb.eq(80,144.92888),new Jrb.eq(81,145.927180629),new Jrb.eq(82,146.924037176),new Jrb.eq(83,147.924298636),new Jrb.eq(84,148.92324163),new Jrb.eq(85,149.923654158),new Jrb.eq(86,150.923098169),new Jrb.eq(87,151.924071324),new Jrb.eq(88,152.923430858),new Jrb.eq(89,153.924686236),new Jrb.eq(90,154.923500411),new Jrb.eq(91,155.924743749),new Jrb.eq(92,156.924021155),new Jrb.eq(93,157.92541026),new Jrb.eq(94,158.925343135),new Jrb.eq(95,159.927164021),new Jrb.eq(96,160.927566289),new Jrb.eq(97,161.929484803),new Jrb.eq(98,162.930643942),new Jrb.eq(99,163.933347253),new Jrb.eq(100,164.93488),new Jrb.eq(101,165.93805),new Jrb.eq(102,166.94005),new Jrb.eq(103,167.94364),new Jrb.eq(104,168.94622),new Jrb.eq(105,169.95025),new Jrb.eq(106,170.9533)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(74,139.95379),new Jrb.eq(75,140.95119),new Jrb.eq(76,141.946695946),new Jrb.eq(77,142.94383),new Jrb.eq(78,143.93907),new Jrb.eq(79,144.936717),new Jrb.eq(80,145.932720118),new Jrb.eq(81,146.930878496),new Jrb.eq(82,147.927177882),new Jrb.eq(83,148.927333981),new Jrb.eq(84,149.925579728),new Jrb.eq(85,150.92617963),new Jrb.eq(86,151.924713874),new Jrb.eq(87,152.925760865),new Jrb.eq(88,153.924422046),new Jrb.eq(89,154.92574895),new Jrb.eq(90,155.924278273),new Jrb.eq(91,156.925461256),new Jrb.eq(92,157.924404637),new Jrb.eq(93,158.92573566),new Jrb.eq(94,159.925193718),new Jrb.eq(95,160.926929595),new Jrb.eq(96,161.926794731),new Jrb.eq(97,162.928727532),new Jrb.eq(98,163.929171165),new Jrb.eq(99,164.931699828),new Jrb.eq(100,165.932803241),new Jrb.eq(101,166.935649025),new Jrb.eq(102,167.93723),new Jrb.eq(103,168.940303648),new Jrb.eq(104,169.94267),new Jrb.eq(105,170.94648),new Jrb.eq(106,171.94911),new Jrb.eq(107,172.95344)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(75,141.95986),new Jrb.eq(76,142.95469),new Jrb.eq(77,143.95164),new Jrb.eq(78,144.94688),new Jrb.eq(79,145.9441),new Jrb.eq(80,146.93984),new Jrb.eq(81,147.937269),new Jrb.eq(82,148.933789944),new Jrb.eq(83,149.932760914),new Jrb.eq(84,150.931680791),new Jrb.eq(85,151.931740598),new Jrb.eq(86,152.930194506),new Jrb.eq(87,153.930596268),new Jrb.eq(88,154.929079084),new Jrb.eq(89,155.929001869),new Jrb.eq(90,156.928188059),new Jrb.eq(91,157.92894573),new Jrb.eq(92,158.927708537),new Jrb.eq(93,159.928725679),new Jrb.eq(94,160.927851662),new Jrb.eq(95,161.92909242),new Jrb.eq(96,162.928730286),new Jrb.eq(97,163.930230577),new Jrb.eq(98,164.930319169),new Jrb.eq(99,165.932281267),new Jrb.eq(100,166.933126195),new Jrb.eq(101,167.935496424),new Jrb.eq(102,168.936868306),new Jrb.eq(103,169.939614951),new Jrb.eq(104,170.941461227),new Jrb.eq(105,171.94482),new Jrb.eq(106,172.94729),new Jrb.eq(107,173.95115),new Jrb.eq(108,174.95405)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(76,143.96059),new Jrb.eq(77,144.95746),new Jrb.eq(78,145.95212),new Jrb.eq(79,146.94931),new Jrb.eq(80,147.94444),new Jrb.eq(81,148.942780527),new Jrb.eq(82,149.937171034),new Jrb.eq(83,150.93746),new Jrb.eq(84,151.935078452),new Jrb.eq(85,152.935093125),new Jrb.eq(86,153.932777294),new Jrb.eq(87,154.933204273),new Jrb.eq(88,155.931015001),new Jrb.eq(89,156.931945517),new Jrb.eq(90,157.929912),new Jrb.eq(91,158.930680718),new Jrb.eq(92,159.929078924),new Jrb.eq(93,160.930001348),new Jrb.eq(94,161.928774923),new Jrb.eq(95,162.930029273),new Jrb.eq(96,163.929196996),new Jrb.eq(97,164.9307228),new Jrb.eq(98,165.93028997),new Jrb.eq(99,166.932045448),new Jrb.eq(100,167.932367781),new Jrb.eq(101,168.934588082),new Jrb.eq(102,169.935460334),new Jrb.eq(103,170.938025885),new Jrb.eq(104,171.939352149),new Jrb.eq(105,172.9424),new Jrb.eq(106,173.94434),new Jrb.eq(107,174.94793),new Jrb.eq(108,175.95029),new Jrb.eq(109,176.95437)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(77,145.966495),new Jrb.eq(78,146.961081),new Jrb.eq(79,147.95755),new Jrb.eq(80,148.95265),new Jrb.eq(81,149.94967),new Jrb.eq(82,150.944842),new Jrb.eq(83,151.9443),new Jrb.eq(84,152.942027631),new Jrb.eq(85,153.940832325),new Jrb.eq(86,154.939191562),new Jrb.eq(87,155.939006895),new Jrb.eq(88,156.936756069),new Jrb.eq(89,157.936996),new Jrb.eq(90,158.934808966),new Jrb.eq(91,159.935090772),new Jrb.eq(92,160.933398042),new Jrb.eq(93,161.933970147),new Jrb.eq(94,162.932647648),new Jrb.eq(95,163.933450972),new Jrb.eq(96,164.932432463),new Jrb.eq(97,165.933553133),new Jrb.eq(98,166.932848844),new Jrb.eq(99,167.934170375),new Jrb.eq(100,168.934211117),new Jrb.eq(101,169.935797877),new Jrb.eq(102,170.936425817),new Jrb.eq(103,171.938396118),new Jrb.eq(104,172.939600336),new Jrb.eq(105,173.942164618),new Jrb.eq(106,174.943832897),new Jrb.eq(107,175.946991412),new Jrb.eq(108,176.94904),new Jrb.eq(109,177.95264),new Jrb.eq(110,178.95534)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(78,147.96676),new Jrb.eq(79,148.96348),new Jrb.eq(80,149.95799),new Jrb.eq(81,150.954657965),new Jrb.eq(82,151.950167),new Jrb.eq(83,152.94921),new Jrb.eq(84,153.945651145),new Jrb.eq(85,154.945792),new Jrb.eq(86,155.942847109),new Jrb.eq(87,156.94265865),new Jrb.eq(88,157.939857897),new Jrb.eq(89,158.940153735),new Jrb.eq(90,159.93756),new Jrb.eq(91,160.937357719),new Jrb.eq(92,161.93575),new Jrb.eq(93,162.936265492),new Jrb.eq(94,163.93452),new Jrb.eq(95,164.935397592),new Jrb.eq(96,165.933879623),new Jrb.eq(97,166.934946862),new Jrb.eq(98,167.933894465),new Jrb.eq(99,168.93518712),new Jrb.eq(100,169.934758652),new Jrb.eq(101,170.936322297),new Jrb.eq(102,171.936377696),new Jrb.eq(103,172.938206756),new Jrb.eq(104,173.938858101),new Jrb.eq(105,174.941272494),new Jrb.eq(106,175.942568409),new Jrb.eq(107,176.945257126),new Jrb.eq(108,177.946643396),new Jrb.eq(109,178.95017),new Jrb.eq(110,179.95233),new Jrb.eq(111,180.95615)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(79,149.972668),new Jrb.eq(80,150.967147),new Jrb.eq(81,151.96361),new Jrb.eq(82,152.95869),new Jrb.eq(83,153.9571),new Jrb.eq(84,154.953641324),new Jrb.eq(85,155.952907),new Jrb.eq(86,156.950101536),new Jrb.eq(87,157.948577981),new Jrb.eq(88,158.946615113),new Jrb.eq(89,159.945383),new Jrb.eq(90,160.943047504),new Jrb.eq(91,161.943222),new Jrb.eq(92,162.941203796),new Jrb.eq(93,163.941215),new Jrb.eq(94,164.939605886),new Jrb.eq(95,165.939762646),new Jrb.eq(96,166.938307056),new Jrb.eq(97,167.938698576),new Jrb.eq(98,168.937648757),new Jrb.eq(99,169.93847219),new Jrb.eq(100,170.937909903),new Jrb.eq(101,171.939082239),new Jrb.eq(102,172.938926901),new Jrb.eq(103,173.940333522),new Jrb.eq(104,174.940767904),new Jrb.eq(105,175.942682399),new Jrb.eq(106,176.943754987),new Jrb.eq(107,177.945951366),new Jrb.eq(108,178.947324216),new Jrb.eq(109,179.949879968),new Jrb.eq(110,180.95197),new Jrb.eq(111,181.95521),new Jrb.eq(112,182.95757),new Jrb.eq(113,183.96117)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(82,153.96425),new Jrb.eq(83,154.96276),new Jrb.eq(84,155.959247),new Jrb.eq(85,156.958127),new Jrb.eq(86,157.95405528),new Jrb.eq(87,158.954003),new Jrb.eq(88,159.950713588),new Jrb.eq(89,160.950330852),new Jrb.eq(90,161.947202977),new Jrb.eq(91,162.947057),new Jrb.eq(92,163.944422),new Jrb.eq(93,164.94454),new Jrb.eq(94,165.94225),new Jrb.eq(95,166.9426),new Jrb.eq(96,167.94063),new Jrb.eq(97,168.941158567),new Jrb.eq(98,169.93965),new Jrb.eq(99,170.94049),new Jrb.eq(100,171.93945798),new Jrb.eq(101,172.94065),new Jrb.eq(102,173.940040159),new Jrb.eq(103,174.941502991),new Jrb.eq(104,175.941401828),new Jrb.eq(105,176.943220013),new Jrb.eq(106,177.943697732),new Jrb.eq(107,178.945815073),new Jrb.eq(108,179.94654876),new Jrb.eq(109,180.949099124),new Jrb.eq(110,181.950552893),new Jrb.eq(111,182.953531012),new Jrb.eq(112,183.95544788),new Jrb.eq(113,184.95878),new Jrb.eq(114,185.96092)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(83,155.971689),new Jrb.eq(84,156.968145),new Jrb.eq(85,157.966368),new Jrb.eq(86,158.96232309),new Jrb.eq(87,159.961358),new Jrb.eq(88,160.958372992),new Jrb.eq(89,161.956556553),new Jrb.eq(90,162.95431665),new Jrb.eq(91,163.95357),new Jrb.eq(92,164.950817),new Jrb.eq(93,165.95047),new Jrb.eq(94,166.948639),new Jrb.eq(95,167.947787),new Jrb.eq(96,168.94592),new Jrb.eq(97,169.94609),new Jrb.eq(98,170.94446),new Jrb.eq(99,171.944739818),new Jrb.eq(100,172.94459),new Jrb.eq(101,173.944167937),new Jrb.eq(102,174.94365),new Jrb.eq(103,175.944740551),new Jrb.eq(104,176.944471766),new Jrb.eq(105,177.945750349),new Jrb.eq(106,178.945934113),new Jrb.eq(107,179.947465655),new Jrb.eq(108,180.947996346),new Jrb.eq(109,181.950152414),new Jrb.eq(110,182.951373188),new Jrb.eq(111,183.954009331),new Jrb.eq(112,184.955559086),new Jrb.eq(113,185.9585501),new Jrb.eq(114,186.96041),new Jrb.eq(115,187.96371)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(84,157.973939),new Jrb.eq(85,158.97228),new Jrb.eq(86,159.968369),new Jrb.eq(87,160.967089),new Jrb.eq(88,161.962750303),new Jrb.eq(89,162.962532),new Jrb.eq(90,163.95898381),new Jrb.eq(91,164.958335962),new Jrb.eq(92,165.955019896),new Jrb.eq(93,166.954672),new Jrb.eq(94,167.951863),new Jrb.eq(95,168.951759),new Jrb.eq(96,169.948473988),new Jrb.eq(97,170.94946),new Jrb.eq(98,171.948228837),new Jrb.eq(99,172.948884),new Jrb.eq(100,173.94616),new Jrb.eq(101,174.94677),new Jrb.eq(102,175.94559),new Jrb.eq(103,176.94662),new Jrb.eq(104,177.945848364),new Jrb.eq(105,178.947071733),new Jrb.eq(106,179.946705734),new Jrb.eq(107,180.948198054),new Jrb.eq(108,181.948205519),new Jrb.eq(109,182.950224458),new Jrb.eq(110,183.950932553),new Jrb.eq(111,184.953420586),new Jrb.eq(112,185.954362204),new Jrb.eq(113,186.957158365),new Jrb.eq(114,187.958486954),new Jrb.eq(115,188.96191222),new Jrb.eq(116,189.963179541)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(85,159.981485),new Jrb.eq(86,160.977661),new Jrb.eq(87,161.975707),new Jrb.eq(88,162.971375872),new Jrb.eq(89,163.970319),new Jrb.eq(90,164.967050268),new Jrb.eq(91,165.965211372),new Jrb.eq(92,166.962564),new Jrb.eq(93,167.961609),new Jrb.eq(94,168.95883),new Jrb.eq(95,169.958163),new Jrb.eq(96,170.955547),new Jrb.eq(97,171.955285),new Jrb.eq(98,172.953062),new Jrb.eq(99,173.952114),new Jrb.eq(100,174.951393),new Jrb.eq(101,175.95157),new Jrb.eq(102,176.95027),new Jrb.eq(103,177.950851081),new Jrb.eq(104,178.949981038),new Jrb.eq(105,179.95078768),new Jrb.eq(106,180.950064596),new Jrb.eq(107,181.951211444),new Jrb.eq(108,182.950821349),new Jrb.eq(109,183.952524289),new Jrb.eq(110,184.952955747),new Jrb.eq(111,185.954986529),new Jrb.eq(112,186.955750787),new Jrb.eq(113,187.958112287),new Jrb.eq(114,188.959228359),new Jrb.eq(115,189.961816139),new Jrb.eq(116,190.963123592),new Jrb.eq(117,191.96596)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(86,161.983819),new Jrb.eq(87,162.982048),new Jrb.eq(88,163.977927),new Jrb.eq(89,164.976475),new Jrb.eq(90,165.971934911),new Jrb.eq(91,166.971554),new Jrb.eq(92,167.967832911),new Jrb.eq(93,168.967076205),new Jrb.eq(94,169.963569716),new Jrb.eq(95,170.96304),new Jrb.eq(96,171.960078),new Jrb.eq(97,172.959791),new Jrb.eq(98,173.956307704),new Jrb.eq(99,174.95708),new Jrb.eq(100,175.953757941),new Jrb.eq(101,176.955045),new Jrb.eq(102,177.953348225),new Jrb.eq(103,178.953951),new Jrb.eq(104,179.952308241),new Jrb.eq(105,180.953274494),new Jrb.eq(106,181.952186222),new Jrb.eq(107,182.95311),new Jrb.eq(108,183.952490808),new Jrb.eq(109,184.954043023),new Jrb.eq(110,185.953838355),new Jrb.eq(111,186.955747928),new Jrb.eq(112,187.955835993),new Jrb.eq(113,188.958144866),new Jrb.eq(114,189.95844521),new Jrb.eq(115,190.960927951),new Jrb.eq(116,191.961479047),new Jrb.eq(117,192.964148083),new Jrb.eq(118,193.965179314),new Jrb.eq(119,194.968123889),new Jrb.eq(120,195.96962255)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(88,164.98758),new Jrb.eq(89,165.985506),new Jrb.eq(90,166.980951577),new Jrb.eq(91,167.979966),new Jrb.eq(92,168.976390868),new Jrb.eq(93,169.974441697),new Jrb.eq(94,170.971779),new Jrb.eq(95,171.970643),new Jrb.eq(96,172.967707),new Jrb.eq(97,173.966804),new Jrb.eq(98,174.964279),new Jrb.eq(99,175.963511),new Jrb.eq(100,176.96117),new Jrb.eq(101,177.960084944),new Jrb.eq(102,178.95915),new Jrb.eq(103,179.958555615),new Jrb.eq(104,180.957642156),new Jrb.eq(105,181.958127689),new Jrb.eq(106,182.956814),new Jrb.eq(107,183.957388318),new Jrb.eq(108,184.95659),new Jrb.eq(109,185.957951104),new Jrb.eq(110,186.95736083),new Jrb.eq(111,187.958851962),new Jrb.eq(112,188.958716473),new Jrb.eq(113,189.960592299),new Jrb.eq(114,190.960591191),new Jrb.eq(115,191.962602198),new Jrb.eq(116,192.9629237),new Jrb.eq(117,193.96507561),new Jrb.eq(118,194.9659768),new Jrb.eq(119,195.968379906),new Jrb.eq(120,196.969636496),new Jrb.eq(121,197.97228),new Jrb.eq(122,198.973787159)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(90,167.988035),new Jrb.eq(91,168.986421),new Jrb.eq(92,169.981734918),new Jrb.eq(93,170.981251),new Jrb.eq(94,171.977376138),new Jrb.eq(95,172.976499642),new Jrb.eq(96,173.972811276),new Jrb.eq(97,174.972276),new Jrb.eq(98,175.969),new Jrb.eq(99,176.968453),new Jrb.eq(100,177.964894223),new Jrb.eq(101,178.965475),new Jrb.eq(102,179.962023729),new Jrb.eq(103,180.963177),new Jrb.eq(104,181.961267637),new Jrb.eq(105,182.961729),new Jrb.eq(106,183.959851685),new Jrb.eq(107,184.960753782),new Jrb.eq(108,185.959432346),new Jrb.eq(109,186.960697),new Jrb.eq(110,187.959395697),new Jrb.eq(111,188.9608319),new Jrb.eq(112,189.959930073),new Jrb.eq(113,190.961684653),new Jrb.eq(114,191.961035158),new Jrb.eq(115,192.962984504),new Jrb.eq(116,193.962663581),new Jrb.eq(117,194.964774449),new Jrb.eq(118,195.964934884),new Jrb.eq(119,196.967323401),new Jrb.eq(120,197.967876009),new Jrb.eq(121,198.970576213),new Jrb.eq(122,199.971423885),new Jrb.eq(123,200.974496467),new Jrb.eq(124,201.97574)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(92,170.991183),new Jrb.eq(93,171.990109),new Jrb.eq(94,172.986398138),new Jrb.eq(95,173.984325861),new Jrb.eq(96,174.981552),new Jrb.eq(97,175.980269),new Jrb.eq(98,176.977215),new Jrb.eq(99,177.975975),new Jrb.eq(100,178.973412),new Jrb.eq(101,179.972396),new Jrb.eq(102,180.969948),new Jrb.eq(103,181.968621416),new Jrb.eq(104,182.96762),new Jrb.eq(105,183.966776046),new Jrb.eq(106,184.965806956),new Jrb.eq(107,185.965997671),new Jrb.eq(108,186.964562),new Jrb.eq(109,187.965321662),new Jrb.eq(110,188.9642243),new Jrb.eq(111,189.964698757),new Jrb.eq(112,190.963649239),new Jrb.eq(113,191.964810107),new Jrb.eq(114,192.964131745),new Jrb.eq(115,193.96533889),new Jrb.eq(116,194.965017928),new Jrb.eq(117,195.966551315),new Jrb.eq(118,196.966551609),new Jrb.eq(119,197.968225244),new Jrb.eq(120,198.968748016),new Jrb.eq(121,199.970717886),new Jrb.eq(122,200.971640839),new Jrb.eq(123,201.973788431),new Jrb.eq(124,202.975137256),new Jrb.eq(125,203.977705),new Jrb.eq(126,204.97961)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(95,174.991411),new Jrb.eq(96,175.987413248),new Jrb.eq(97,176.986336874),new Jrb.eq(98,177.982476325),new Jrb.eq(99,178.981783),new Jrb.eq(100,179.978322),new Jrb.eq(101,180.977806),new Jrb.eq(102,181.97393546),new Jrb.eq(103,182.974561),new Jrb.eq(104,183.970705219),new Jrb.eq(105,184.971983),new Jrb.eq(106,185.969460021),new Jrb.eq(107,186.969785),new Jrb.eq(108,187.967511693),new Jrb.eq(109,188.968733187),new Jrb.eq(110,189.966958568),new Jrb.eq(111,190.96706311),new Jrb.eq(112,191.965921572),new Jrb.eq(113,192.966644169),new Jrb.eq(114,193.965381832),new Jrb.eq(115,194.966638981),new Jrb.eq(116,195.965814846),new Jrb.eq(117,196.967195333),new Jrb.eq(118,197.96675183),new Jrb.eq(119,198.968262489),new Jrb.eq(120,199.968308726),new Jrb.eq(121,200.970285275),new Jrb.eq(122,201.970625604),new Jrb.eq(123,202.972857096),new Jrb.eq(124,203.97347564),new Jrb.eq(125,204.976056104),new Jrb.eq(126,205.977498672),new Jrb.eq(127,206.982577025),new Jrb.eq(128,207.98594)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(96,176.996881),new Jrb.eq(97,177.994637),new Jrb.eq(98,178.991466),new Jrb.eq(99,179.990194),new Jrb.eq(100,180.986904),new Jrb.eq(101,181.98561),new Jrb.eq(102,182.982697),new Jrb.eq(103,183.98176),new Jrb.eq(104,184.9791),new Jrb.eq(105,185.977549881),new Jrb.eq(106,186.97617),new Jrb.eq(107,187.97592),new Jrb.eq(108,188.974290451),new Jrb.eq(109,189.974473379),new Jrb.eq(110,190.972261952),new Jrb.eq(111,191.972770785),new Jrb.eq(112,192.970548),new Jrb.eq(113,193.971053),new Jrb.eq(114,194.96965),new Jrb.eq(115,195.970515),new Jrb.eq(116,196.9695362),new Jrb.eq(117,197.970466294),new Jrb.eq(118,198.969813837),new Jrb.eq(119,199.970945394),new Jrb.eq(120,200.97080377),new Jrb.eq(121,201.972090569),new Jrb.eq(122,202.972329088),new Jrb.eq(123,203.973848646),new Jrb.eq(124,204.97441227),new Jrb.eq(125,205.976095321),new Jrb.eq(126,206.977407908),new Jrb.eq(127,207.982004653),new Jrb.eq(128,208.985349125),new Jrb.eq(129,209.990065574)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(99,180.996714),new Jrb.eq(100,181.992676101),new Jrb.eq(101,182.99193),new Jrb.eq(102,183.988198),new Jrb.eq(103,184.98758),new Jrb.eq(104,185.983485388),new Jrb.eq(105,186.98403),new Jrb.eq(106,187.979869108),new Jrb.eq(107,188.98088),new Jrb.eq(108,189.978180008),new Jrb.eq(109,190.9782),new Jrb.eq(110,191.975719811),new Jrb.eq(111,192.97608),new Jrb.eq(112,193.974648056),new Jrb.eq(113,194.975920279),new Jrb.eq(114,195.97271),new Jrb.eq(115,196.97338),new Jrb.eq(116,197.97198),new Jrb.eq(117,198.972909384),new Jrb.eq(118,199.97181556),new Jrb.eq(119,200.972846589),new Jrb.eq(120,201.972143786),new Jrb.eq(121,202.973375491),new Jrb.eq(122,203.973028761),new Jrb.eq(123,204.974467112),new Jrb.eq(124,205.974449002),new Jrb.eq(125,206.975880605),new Jrb.eq(126,207.97663585),new Jrb.eq(127,208.981074801),new Jrb.eq(128,209.984173129),new Jrb.eq(129,210.988731474),new Jrb.eq(130,211.991887495),new Jrb.eq(131,212.9965),new Jrb.eq(132,213.999798147)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(102,184.997708),new Jrb.eq(103,185.99648),new Jrb.eq(104,186.993458),new Jrb.eq(105,187.992173),new Jrb.eq(106,188.989505),new Jrb.eq(107,189.987520007),new Jrb.eq(108,190.986053),new Jrb.eq(109,191.985368),new Jrb.eq(110,192.983662229),new Jrb.eq(111,193.983430186),new Jrb.eq(112,194.98112697),new Jrb.eq(113,195.981236107),new Jrb.eq(114,196.978934287),new Jrb.eq(115,197.979024396),new Jrb.eq(116,198.977576953),new Jrb.eq(117,199.978141983),new Jrb.eq(118,200.976970721),new Jrb.eq(119,201.977674504),new Jrb.eq(120,202.976868118),new Jrb.eq(121,203.977805161),new Jrb.eq(122,204.977374688),new Jrb.eq(123,205.978482854),new Jrb.eq(124,206.978455217),new Jrb.eq(125,207.979726699),new Jrb.eq(126,208.980383241),new Jrb.eq(127,209.984104944),new Jrb.eq(128,210.987258139),new Jrb.eq(129,211.991271542),new Jrb.eq(130,212.994374836),new Jrb.eq(131,213.998698664),new Jrb.eq(132,215.001832349),new Jrb.eq(133,216.006199)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(106,189.994293888),new Jrb.eq(107,190.994653),new Jrb.eq(108,191.99033039),new Jrb.eq(109,192.991102),new Jrb.eq(110,193.988284107),new Jrb.eq(111,194.988045),new Jrb.eq(112,195.985469432),new Jrb.eq(113,196.985567),new Jrb.eq(114,197.984024384),new Jrb.eq(115,198.985044507),new Jrb.eq(116,199.981735),new Jrb.eq(117,200.982209),new Jrb.eq(118,201.980704),new Jrb.eq(119,202.981412863),new Jrb.eq(120,203.980307113),new Jrb.eq(121,204.981165396),new Jrb.eq(122,205.980465241),new Jrb.eq(123,206.981578228),new Jrb.eq(124,207.981231059),new Jrb.eq(125,208.982415788),new Jrb.eq(126,209.982857396),new Jrb.eq(127,210.986636869),new Jrb.eq(128,211.988851755),new Jrb.eq(129,212.992842522),new Jrb.eq(130,213.995185949),new Jrb.eq(131,214.999414609),new Jrb.eq(132,216.001905198),new Jrb.eq(133,217.006253),new Jrb.eq(134,218.008965773)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(108,193.000188),new Jrb.eq(109,193.997973),new Jrb.eq(110,194.996554),new Jrb.eq(111,195.995702),new Jrb.eq(112,196.993891293),new Jrb.eq(113,197.99343368),new Jrb.eq(114,198.991008569),new Jrb.eq(115,199.990920883),new Jrb.eq(116,200.988486908),new Jrb.eq(117,201.988448629),new Jrb.eq(118,202.986847216),new Jrb.eq(119,203.987261559),new Jrb.eq(120,204.986036352),new Jrb.eq(121,205.986599242),new Jrb.eq(122,206.985775861),new Jrb.eq(123,207.986582508),new Jrb.eq(124,208.986158678),new Jrb.eq(125,209.987131308),new Jrb.eq(126,210.987480806),new Jrb.eq(127,211.990734657),new Jrb.eq(128,212.99292115),new Jrb.eq(129,213.996356412),new Jrb.eq(130,214.998641245),new Jrb.eq(131,216.002408839),new Jrb.eq(132,217.004709619),new Jrb.eq(133,218.008681458),new Jrb.eq(134,219.011296478),new Jrb.eq(135,220.015301),new Jrb.eq(136,221.01814),new Jrb.eq(137,222.02233),new Jrb.eq(138,223.02534)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(110,196.001117268),new Jrb.eq(111,197.001661),new Jrb.eq(112,197.998779978),new Jrb.eq(113,198.998309),new Jrb.eq(114,199.995634148),new Jrb.eq(115,200.995535),new Jrb.eq(116,201.993899382),new Jrb.eq(117,202.994765192),new Jrb.eq(118,203.991365),new Jrb.eq(119,204.991668),new Jrb.eq(120,205.99016),new Jrb.eq(121,206.990726826),new Jrb.eq(122,207.989631237),new Jrb.eq(123,208.990376634),new Jrb.eq(124,209.989679862),new Jrb.eq(125,210.99058541),new Jrb.eq(126,211.990688899),new Jrb.eq(127,212.993868354),new Jrb.eq(128,213.995346275),new Jrb.eq(129,214.998729195),new Jrb.eq(130,216.000258153),new Jrb.eq(131,217.003914555),new Jrb.eq(132,218.005586315),new Jrb.eq(133,219.009474831),new Jrb.eq(134,220.011384149),new Jrb.eq(135,221.015455),new Jrb.eq(136,222.017570472),new Jrb.eq(137,223.02179),new Jrb.eq(138,224.02409),new Jrb.eq(139,225.02844),new Jrb.eq(140,226.03089),new Jrb.eq(141,227.035407),new Jrb.eq(142,228.038084)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(113,200.006499),new Jrb.eq(114,201.00458692),new Jrb.eq(115,202.00396885),new Jrb.eq(116,203.001423829),new Jrb.eq(117,204.001221209),new Jrb.eq(118,204.998663961),new Jrb.eq(119,205.998486886),new Jrb.eq(120,206.996859385),new Jrb.eq(121,207.997133849),new Jrb.eq(122,208.995915421),new Jrb.eq(123,209.996398327),new Jrb.eq(124,210.995529332),new Jrb.eq(125,211.996194988),new Jrb.eq(126,212.996174845),new Jrb.eq(127,213.99895474),new Jrb.eq(128,215.000326029),new Jrb.eq(129,216.003187873),new Jrb.eq(130,217.004616452),new Jrb.eq(131,218.007563326),new Jrb.eq(132,219.009240843),new Jrb.eq(133,220.012312978),new Jrb.eq(134,221.014245654),new Jrb.eq(135,222.017543957),new Jrb.eq(136,223.019730712),new Jrb.eq(137,224.023235513),new Jrb.eq(138,225.025606914),new Jrb.eq(139,226.029343423),new Jrb.eq(140,227.031833167),new Jrb.eq(141,228.034776087),new Jrb.eq(142,229.038426),new Jrb.eq(143,230.04251),new Jrb.eq(144,231.045407),new Jrb.eq(145,232.049654)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(115,203.00921),new Jrb.eq(116,204.006434513),new Jrb.eq(117,205.006187),new Jrb.eq(118,206.004463814),new Jrb.eq(119,207.005176607),new Jrb.eq(120,208.001776),new Jrb.eq(121,209.001944),new Jrb.eq(122,210.000446),new Jrb.eq(123,211.000893996),new Jrb.eq(124,211.999783492),new Jrb.eq(125,213.000345847),new Jrb.eq(126,214.000091141),new Jrb.eq(127,215.002704195),new Jrb.eq(128,216.003518402),new Jrb.eq(129,217.00630601),new Jrb.eq(130,218.007123948),new Jrb.eq(131,219.010068787),new Jrb.eq(132,220.011014669),new Jrb.eq(133,221.013907762),new Jrb.eq(134,222.01536182),new Jrb.eq(135,223.01849714),new Jrb.eq(136,224.020202004),new Jrb.eq(137,225.023604463),new Jrb.eq(138,226.025402555),new Jrb.eq(139,227.029170677),new Jrb.eq(140,228.031064101),new Jrb.eq(141,229.034820309),new Jrb.eq(142,230.037084774),new Jrb.eq(143,231.04122),new Jrb.eq(144,232.043693),new Jrb.eq(145,233.047995),new Jrb.eq(146,234.050547)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(118,207.012469754),new Jrb.eq(119,208.012112949),new Jrb.eq(120,209.009568736),new Jrb.eq(121,210.009256802),new Jrb.eq(122,211.007648196),new Jrb.eq(123,212.007811441),new Jrb.eq(124,213.006573689),new Jrb.eq(125,214.006893072),new Jrb.eq(126,215.006450832),new Jrb.eq(127,216.008721268),new Jrb.eq(128,217.009332676),new Jrb.eq(129,218.011625045),new Jrb.eq(130,219.012404918),new Jrb.eq(131,220.014752105),new Jrb.eq(132,221.015575746),new Jrb.eq(133,222.017828852),new Jrb.eq(134,223.01912603),new Jrb.eq(135,224.021708435),new Jrb.eq(136,225.023220576),new Jrb.eq(137,226.026089848),new Jrb.eq(138,227.027746979),new Jrb.eq(139,228.031014825),new Jrb.eq(140,229.032930871),new Jrb.eq(141,230.036025144),new Jrb.eq(142,231.038551503),new Jrb.eq(143,232.042022474),new Jrb.eq(144,233.04455),new Jrb.eq(145,234.04842),new Jrb.eq(146,235.051102),new Jrb.eq(147,236.055178)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(120,210.015711883),new Jrb.eq(121,211.016306912),new Jrb.eq(122,212.012916),new Jrb.eq(123,213.012962),new Jrb.eq(124,214.011451),new Jrb.eq(125,215.011726597),new Jrb.eq(126,216.011050963),new Jrb.eq(127,217.013066169),new Jrb.eq(128,218.013267744),new Jrb.eq(129,219.015521253),new Jrb.eq(130,220.015733126),new Jrb.eq(131,221.018171499),new Jrb.eq(132,222.018454131),new Jrb.eq(133,223.020795153),new Jrb.eq(134,224.02145925),new Jrb.eq(135,225.023941441),new Jrb.eq(136,226.024890681),new Jrb.eq(137,227.027698859),new Jrb.eq(138,228.028731348),new Jrb.eq(139,229.03175534),new Jrb.eq(140,230.033126574),new Jrb.eq(141,231.03629706),new Jrb.eq(142,232.03805036),new Jrb.eq(143,233.041576923),new Jrb.eq(144,234.043595497),new Jrb.eq(145,235.04750442),new Jrb.eq(146,236.04971),new Jrb.eq(147,237.053894),new Jrb.eq(148,238.056243)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(122,213.021183209),new Jrb.eq(123,214.02073923),new Jrb.eq(124,215.019097612),new Jrb.eq(125,216.019109649),new Jrb.eq(126,217.018288571),new Jrb.eq(127,218.020007906),new Jrb.eq(128,219.019880348),new Jrb.eq(129,220.021876493),new Jrb.eq(130,221.021863742),new Jrb.eq(131,222.023726),new Jrb.eq(132,223.023963748),new Jrb.eq(133,224.025614854),new Jrb.eq(134,225.026115172),new Jrb.eq(135,226.02793275),new Jrb.eq(136,227.028793151),new Jrb.eq(137,228.031036942),new Jrb.eq(138,229.032088601),new Jrb.eq(139,230.034532562),new Jrb.eq(140,231.035878898),new Jrb.eq(141,232.03858172),new Jrb.eq(142,233.040240235),new Jrb.eq(143,234.043302325),new Jrb.eq(144,235.045436759),new Jrb.eq(145,236.048675176),new Jrb.eq(146,237.05113943),new Jrb.eq(147,238.054497046),new Jrb.eq(148,239.05713),new Jrb.eq(149,240.06098)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(126,218.023487),new Jrb.eq(127,219.024915423),new Jrb.eq(128,220.024712),new Jrb.eq(129,221.026351),new Jrb.eq(130,222.02607),new Jrb.eq(131,223.027722956),new Jrb.eq(132,224.027590139),new Jrb.eq(133,225.029384369),new Jrb.eq(134,226.02933975),new Jrb.eq(135,227.031140069),new Jrb.eq(136,228.031366357),new Jrb.eq(137,229.033496137),new Jrb.eq(138,230.033927392),new Jrb.eq(139,231.036289158),new Jrb.eq(140,232.03714628),new Jrb.eq(141,233.039628196),new Jrb.eq(142,234.040945606),new Jrb.eq(143,235.043923062),new Jrb.eq(144,236.045561897),new Jrb.eq(145,237.048723955),new Jrb.eq(146,238.050782583),new Jrb.eq(147,239.054287777),new Jrb.eq(148,240.056585734),new Jrb.eq(149,241.06033),new Jrb.eq(150,242.062925)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(132,225.033899689),new Jrb.eq(133,226.035129),new Jrb.eq(134,227.034958261),new Jrb.eq(135,228.03618),new Jrb.eq(136,229.036246866),new Jrb.eq(137,230.037812591),new Jrb.eq(138,231.038233161),new Jrb.eq(139,232.040099),new Jrb.eq(140,233.04073235),new Jrb.eq(141,234.042888556),new Jrb.eq(142,235.044055876),new Jrb.eq(143,236.046559724),new Jrb.eq(144,237.048167253),new Jrb.eq(145,238.050940464),new Jrb.eq(146,239.052931399),new Jrb.eq(147,240.056168828),new Jrb.eq(148,241.058246266),new Jrb.eq(149,242.061635),new Jrb.eq(150,243.064273),new Jrb.eq(151,244.06785)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(134,228.038727686),new Jrb.eq(135,229.040138934),new Jrb.eq(136,230.039645603),new Jrb.eq(137,231.041258),new Jrb.eq(138,232.041179445),new Jrb.eq(139,233.04298757),new Jrb.eq(140,234.043304681),new Jrb.eq(141,235.0452815),new Jrb.eq(142,236.046048088),new Jrb.eq(143,237.048403774),new Jrb.eq(144,238.0495534),new Jrb.eq(145,239.052156519),new Jrb.eq(146,240.05380746),new Jrb.eq(147,241.056845291),new Jrb.eq(148,242.058736847),new Jrb.eq(149,243.061997013),new Jrb.eq(150,244.06419765),new Jrb.eq(151,245.067738657),new Jrb.eq(152,246.070198429),new Jrb.eq(153,247.07407)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(136,231.04556),new Jrb.eq(137,232.04659),new Jrb.eq(138,233.046472),new Jrb.eq(139,234.047794),new Jrb.eq(140,235.048029),new Jrb.eq(141,236.049569),new Jrb.eq(142,237.049970748),new Jrb.eq(143,238.051977839),new Jrb.eq(144,239.053018481),new Jrb.eq(145,240.055287826),new Jrb.eq(146,241.056822944),new Jrb.eq(147,242.059543039),new Jrb.eq(148,243.061372686),new Jrb.eq(149,244.064279429),new Jrb.eq(150,245.066445398),new Jrb.eq(151,246.069768438),new Jrb.eq(152,247.072086),new Jrb.eq(153,248.075745),new Jrb.eq(154,249.07848)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(137,233.0508),new Jrb.eq(138,234.05024),new Jrb.eq(139,235.051591),new Jrb.eq(140,236.051405),new Jrb.eq(141,237.052891),new Jrb.eq(142,238.053016298),new Jrb.eq(143,239.054951),new Jrb.eq(144,240.055519046),new Jrb.eq(145,241.057646736),new Jrb.eq(146,242.058829326),new Jrb.eq(147,243.061382249),new Jrb.eq(148,244.062746349),new Jrb.eq(149,245.065485586),new Jrb.eq(150,246.067217551),new Jrb.eq(151,247.070346811),new Jrb.eq(152,248.072342247),new Jrb.eq(153,249.075947062),new Jrb.eq(154,250.078350687),new Jrb.eq(155,251.082277873),new Jrb.eq(156,252.08487)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(138,235.05658),new Jrb.eq(139,236.05733),new Jrb.eq(140,237.057127),new Jrb.eq(141,238.058266),new Jrb.eq(142,239.058362),new Jrb.eq(143,240.059749),new Jrb.eq(144,241.060223),new Jrb.eq(145,242.06205),new Jrb.eq(146,243.06300157),new Jrb.eq(147,244.065167882),new Jrb.eq(148,245.066355386),new Jrb.eq(149,246.068666836),new Jrb.eq(150,247.070298533),new Jrb.eq(151,248.07308),new Jrb.eq(152,249.074979937),new Jrb.eq(153,250.078310529),new Jrb.eq(154,251.08075344),new Jrb.eq(155,252.084303),new Jrb.eq(156,253.08688),new Jrb.eq(157,254.0906)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(139,237.06207),new Jrb.eq(140,238.06141),new Jrb.eq(141,239.062579),new Jrb.eq(142,240.062295),new Jrb.eq(143,241.063716),new Jrb.eq(144,242.063688713),new Jrb.eq(145,243.065421),new Jrb.eq(146,244.06599039),new Jrb.eq(147,245.068039),new Jrb.eq(148,246.068798807),new Jrb.eq(149,247.070992043),new Jrb.eq(150,248.07217808),new Jrb.eq(151,249.074846818),new Jrb.eq(152,250.076399951),new Jrb.eq(153,251.079580056),new Jrb.eq(154,252.081619582),new Jrb.eq(155,253.085126791),new Jrb.eq(156,254.087316198),new Jrb.eq(157,255.091039),new Jrb.eq(158,256.09344)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(141,240.06892),new Jrb.eq(142,241.068662),new Jrb.eq(143,242.069699),new Jrb.eq(144,243.069631),new Jrb.eq(145,244.070969),new Jrb.eq(146,245.071317),new Jrb.eq(147,246.072965),new Jrb.eq(148,247.07365),new Jrb.eq(149,248.075458),new Jrb.eq(150,249.076405),new Jrb.eq(151,250.078654),new Jrb.eq(152,251.079983592),new Jrb.eq(153,252.082972247),new Jrb.eq(154,253.084817974),new Jrb.eq(155,254.088016026),new Jrb.eq(156,255.090266386),new Jrb.eq(157,256.093592),new Jrb.eq(158,257.095979)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(142,242.07343),new Jrb.eq(143,243.07451),new Jrb.eq(144,244.074077),new Jrb.eq(145,245.075375),new Jrb.eq(146,246.075281634),new Jrb.eq(147,247.076819),new Jrb.eq(148,248.077184411),new Jrb.eq(149,249.079024),new Jrb.eq(150,250.079514759),new Jrb.eq(151,251.081566467),new Jrb.eq(152,252.082460071),new Jrb.eq(153,253.085176259),new Jrb.eq(154,254.086847795),new Jrb.eq(155,255.089955466),new Jrb.eq(156,256.091766522),new Jrb.eq(157,257.095098635),new Jrb.eq(158,258.097069),new Jrb.eq(159,259.100588)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(144,245.081017),new Jrb.eq(145,246.081933),new Jrb.eq(146,247.081804),new Jrb.eq(147,248.082909),new Jrb.eq(148,249.083002),new Jrb.eq(149,250.084488),new Jrb.eq(150,251.084919),new Jrb.eq(151,252.08663),new Jrb.eq(152,253.08728),new Jrb.eq(153,254.089725),new Jrb.eq(154,255.091075196),new Jrb.eq(155,256.094052757),new Jrb.eq(156,257.095534643),new Jrb.eq(157,258.098425321),new Jrb.eq(158,259.100503),new Jrb.eq(159,260.103645)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(147,249.087823),new Jrb.eq(148,250.087493),new Jrb.eq(149,251.08896),new Jrb.eq(150,252.088965909),new Jrb.eq(151,253.090649),new Jrb.eq(152,254.090948746),new Jrb.eq(153,255.093232449),new Jrb.eq(154,256.094275879),new Jrb.eq(155,257.096852778),new Jrb.eq(156,258.0982),new Jrb.eq(157,259.101024),new Jrb.eq(158,260.102636),new Jrb.eq(159,261.105743),new Jrb.eq(160,262.10752)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(148,251.09436),new Jrb.eq(149,252.09533),new Jrb.eq(150,253.095258),new Jrb.eq(151,254.096587),new Jrb.eq(152,255.096769),new Jrb.eq(153,256.098763),new Jrb.eq(154,257.099606),new Jrb.eq(155,258.101883),new Jrb.eq(156,259.10299),new Jrb.eq(157,260.105572),new Jrb.eq(158,261.106941),new Jrb.eq(159,262.109692),new Jrb.eq(160,263.111394)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(149,253.100679),new Jrb.eq(150,254.100166),new Jrb.eq(151,255.101492),new Jrb.eq(152,256.101179573),new Jrb.eq(153,257.103072),new Jrb.eq(154,258.103568),new Jrb.eq(155,259.105628),new Jrb.eq(156,260.106434),new Jrb.eq(157,261.108752),new Jrb.eq(158,262.109918),new Jrb.eq(159,263.11254),new Jrb.eq(160,264.113978)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(150,255.107398),new Jrb.eq(151,256.10811),new Jrb.eq(152,257.107858),new Jrb.eq(153,258.109438),new Jrb.eq(154,259.109721),new Jrb.eq(155,260.111427),new Jrb.eq(156,261.112106),new Jrb.eq(157,262.114153),new Jrb.eq(158,263.115078),new Jrb.eq(159,264.117473),new Jrb.eq(160,265.118659)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(152,258.113151),new Jrb.eq(153,259.114652),new Jrb.eq(154,260.114435447),new Jrb.eq(155,261.116199),new Jrb.eq(156,262.116477),new Jrb.eq(157,263.118313),new Jrb.eq(158,264.118924),new Jrb.eq(159,265.121066),new Jrb.eq(160,266.121928)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(153,260.121803),new Jrb.eq(154,261.1218),new Jrb.eq(155,262.123009),new Jrb.eq(156,263.123146),new Jrb.eq(157,264.12473),new Jrb.eq(158,265.125198),new Jrb.eq(159,266.127009),new Jrb.eq(160,267.12774)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(155,263.12871),new Jrb.eq(156,264.128408258),new Jrb.eq(157,265.130001),new Jrb.eq(158,266.130042),new Jrb.eq(159,267.131774),new Jrb.eq(160,268.132156),new Jrb.eq(161,269.134114)]),BX(nX(Drb.nZ,1),Uub,3,0,[new Jrb.eq(156,265.136567),new Jrb.eq(157,266.13794),new Jrb.eq(158,267.137526),new Jrb.eq(159,268.138816),new Jrb.eq(160,269.139106),new Jrb.eq(161,270.140723),new Jrb.eq(162,271.141229)])]);};Jrb.hq=function hq(a,b){Jrb.gq();var c,d;d=b-a;for(c=0;c<Jrb.fq[a].length;c++)if(Jrb.fq[a][c].b==d)return Jrb.fq[a][c].a;return NaN;};Jrb.iq=function iq(a,b){var c;if(a.b.length!=b.b.length)return a.b.length<b.b.length?-1:1;for(c=0;c<a.b.length;c++)if(a.b[c]!=b.b[c])return a.b[c]<b.b[c]?-1:1;return 0;};Jrb.jq=function jq(a){var b,c,d,e,f,g;c=0;for(e=a.a,f=0,g=e.length;f<g;++f){d=e[f];d&&++c;}a.b=tX(Drb.NY,Qsb,6,c,15,1);c=0;for(b=0;b<a.a.length;b++)a.a[b]&&(a.b[c++]=b);};Jrb.kq=function kq(a,b){var c;Jrb.jq(a);for(c=0;c<a.b.length;c++)if(b[a.b[c]])return true;return false;};Jrb.lq=function lq(a){this.a=tX(Drb.c3,itb,6,a,16,1);};Z3(203,1,{203:1,28:1},Jrb.lq);_.rb=function mq(a){return Jrb.iq(this,a);};Drb.oZ=B5(203);Jrb.qq=function qq(){Jrb.qq=_3;Jrb.pq=BX(nX(Drb.LY,1),Osb,6,15,[0,1.00794,4.0026,6.941,9.0122,10.811,12.011,14.007,15.999,18.998,20.18,22.99,24.305,26.982,28.086,30.974,32.066,35.453,39.948,39.098,40.078,44.956,47.867,50.942,51.996,54.938,55.845,58.933,58.693,63.546,65.39,69.723,72.61,74.922,78.96,79.904,83.8,85.468,87.62,88.906,91.224,92.906,95.94,98.906,101.07,102.91,106.42,107.87,112.41,114.82,118.71,121.76,127.6,126.9,131.29,132.91,137.33,138.91,140.12,140.91,144.24,146.92,150.36,151.96,157.25,158.93,162.5,164.93,167.26,168.93,173.04,174.97,178.49,180.95,183.84,186.21,190.23,192.22,195.08,196.97,200.59,204.38,207.2,208.98,209.98,209.99,222.02,223.02,226.03,227.03,232.04,231.04,238.03,237.05,239.05,241.06,244.06,249.08,252.08,252.08,257.1,258.1,259.1,262.11,267.12,268.13,271.13,270.13,277.15,276.15,Vub,Vub,283.17,285.18,Wub,Wub,293.2,Xub,Xub,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.0141,3.016,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71.0787,156.18828,114.10364,115.0877,103.1447,128.13052,129.11458,57.05182,137.14158,113.15934,113.15934,128.17428,131.19846,147.17646,97.11658,87.0777,101.10458,186.2134,163.17546,99.13246]);Jrb.nq=BX(nX(Drb.LY,1),Osb,6,15,[0,1.007825,4.0026,7.016003,9.012182,11.009305,12,14.003074,15.994915,18.998403,19.992435,22.989767,23.985042,26.98153,27.976927,30.973762,31.97207,34.968852,39.962384,38.963707,39.962591,44.95591,47.947947,50.943962,51.940509,54.938047,55.934939,58.933198,57.935346,62.939598,63.929145,68.92558,73.921177,74.921594,79.91652,78.918336,83.911507,84.911794,87.905619,88.905849,89.904703,92.906377,97.905406,89.92381,101.904348,102.9055,105.903478,106.905092,113.903357,114.90388,119.9022,120.903821,129.906229,126.904473,131.904144,132.905429,137.905232,138.906346,139.905433,140.907647,141.907719,135.92398,151.919729,152.921225,157.924099,158.925342,163.929171,164.930319,165.93029,168.934212,173.938859,174.94077,179.946545,180.947992,183.950928,186.955744,191.961467,192.962917,194.964766,196.966543,201.970617,204.974401,207.976627,208.980374,193.98818,195.99573,199.9957,201.00411,206.0038,210.00923,232.038054,216.01896,238.050784,229.03623,232.041169,237.05005,238.05302,242.06194,240.06228,243.06947,243.07446,248.08275,251.08887,253.09515,257.10295,257.10777,271.13,270.13,277.15,276.15,Vub,Vub,283.17,285.18,Wub,Wub,291.2,Xub,Xub,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.014,3.01605,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]);Jrb.oq=BX(nX(Drb.NY,1),Qsb,6,15,[6,1,7,8]);};Jrb.rq=function rq(a){var b,c;c=a.a;for(b=0;b<a.b.length;b++)c+=a.b[b]*Jrb.nq[a.c[b]];return c;};Jrb.sq=function sq(a){var b,c;b=new Erb.I7();for(c=0;c<a.b.length;c++){Erb.H7(b,(Jrb.Ej(),Jrb.yj)[a.c[c]]);a.b[c]>1&&Erb.G7(b,a.b[c]);}return b.a;};Jrb.tq=function tq(a){var b,c;c=a.d;for(b=0;b<a.b.length;b++)c+=a.b[b]*Jrb.pq[a.c[b]];return c;};Jrb.uq=function uq(a){var b,c,d,e,f,g,h,i,j,k,l;Jrb.ou(a,1);e=tX(Drb.NY,Qsb,6,191,15,1);for(c=0;c<a.q;c++){switch(a.C[c]){case 171:e[1]+=5;e[6]+=3;e[7]+=1;e[8]+=1;break;case 172:e[1]+=12;e[6]+=6;e[7]+=4;e[8]+=1;break;case 173:e[1]+=6;e[6]+=4;e[7]+=2;e[8]+=2;break;case 174:e[1]+=5;e[6]+=4;e[7]+=1;e[8]+=3;break;case 175:e[1]+=5;e[6]+=3;e[7]+=1;e[8]+=1;e[16]+=1;break;case 176:e[1]+=8;e[6]+=5;e[7]+=2;e[8]+=2;break;case 177:e[1]+=7;e[6]+=5;e[7]+=1;e[8]+=3;break;case 178:e[1]+=3;e[6]+=2;e[7]+=1;e[8]+=1;break;case 179:e[1]+=7;e[6]+=6;e[7]+=3;e[8]+=1;break;case 181:case 180:e[1]+=11;e[6]+=6;e[7]+=1;e[8]+=1;break;case 182:e[1]+=12;e[6]+=6;e[7]+=2;e[8]+=1;break;case 183:e[1]+=9;e[6]+=5;e[7]+=1;e[8]+=1;e[16]+=1;break;case 184:e[1]+=9;e[6]+=9;e[7]+=1;e[8]+=1;break;case 185:e[1]+=7;e[6]+=5;e[7]+=1;e[8]+=1;break;case 186:e[1]+=5;e[6]+=3;e[7]+=1;e[8]+=2;break;case 187:e[1]+=7;e[6]+=4;e[7]+=1;e[8]+=2;break;case 188:e[1]+=10;e[6]+=11;e[7]+=2;e[8]+=1;break;case 189:e[1]+=9;e[6]+=9;e[7]+=1;e[8]+=2;break;case 190:e[1]+=9;e[6]+=5;e[7]+=1;e[8]+=1;break;case 1:switch(a.A[c]){case 0:case 1:++e[1];break;case 2:++e[151];break;case 3:++e[152];}break;default:++e[a.C[c]];}}for(d=0;d<a.q;d++)a.C[d]>=171&&a.C[d]<=190?e[1]+=2-Jrb.ao(a,d):e[1]+=Jrb.Wn(a,d);h=0;for(j=1;j<=190;j++)e[j]!=0&&++h;this.b=tX(Drb.NY,Qsb,6,h,15,1);this.c=tX(Drb.NY,Qsb,6,h,15,1);h=0;for(i=0;i<Jrb.oq.length;i++){if(e[Jrb.oq[i]]!=0){this.b[h]=e[Jrb.oq[i]];this.c[h]=Jrb.oq[i];++h;e[Jrb.oq[i]]=0;}}while(true){l='zzz';k=-1;for(g=1;g<=190;g++)if(e[g]>0&&Erb.e7(l,(Jrb.Ej(),Jrb.yj)[g])>0){l=(Jrb.Ej(),Jrb.yj)[g];k=g;}if(k==-1)break;this.b[h]=e[k];this.c[h]=k;++h;e[k]=0;}this.a=0;this.d=0;for(b=0;b<a.f;b++){if(a.C[b]!=1&&a.A[b]!=0){g=a.C[b];f=a.A[b];this.a+=Jrb.hq(g,f)-Jrb.nq[g];this.d+=Jrb.hq(g,f)-Jrb.pq[g];}}};Z3(128,1,{128:1});_.jb=function vq(a){var b;if(a===this)return true;if(!zY(a,128))return false;for(b=0;b<this.b.length;b++)if(this.b[b]!=a.b[b])return false;return true;};_.a=0;_.d=0;Drb.pZ=B5(128);Jrb.wq=function wq(a){switch(a){case 6:return 1;case 53:return 2;case 33:return 3;case 34:return 4;case 35:return 5;case 15:return 6;case 16:return 7;case 17:return 8;case 7:return 9;case 8:return 10;case 9:return 11;}return 0;};Jrb.xq=function xq(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C;Jrb.ou(a,7);B=0;n=tX(Drb.c3,itb,6,a.f,16,1);C=tX(Drb.NY,Qsb,6,a.f,15,1);for(f=0;f<a.g;f++){c=a.D[0][f];d=a.D[1][f];g=a.s[c];h=a.s[d];if(g!=0&&h!=0&&g<0^h<0){n[c]=true;n[d]=true;}C[c]+=Jrb.Qk(a,f);C[d]+=Jrb.Qk(a,f);}for(i=0;i<a.f;i++){B+=a.s[i];if(a.s[i]==1){if(a.C[i]==7){if(!n[i]){if(C[i]<=3){B-=1;a.s[i]=0;a.T=0;if(a.j[i]!=a.e[i]){Jrb.bk(a,Jrb.Fn(a,i,a.e[i]-1));Jrb.ou(a,7);}}else if(i<a.f&&Jrb.Fr(a.p,i)){r=tX(Drb.c3,itb,6,a.f,16,1);s=tX(Drb.c3,itb,6,a.g,16,1);Jrb.on(a,i,true,r,s);for(k=0;k<a.f;k++){if(r[k]&&a.C[k]==7&&a.s[k]==0&&C[k]==2){if(Jrb.Aq(a,s,i,k)){B-=1;break;}}}}}}}else if(a.s[i]<0){if(a.C[i]==6||a.C[i]==7||a.C[i]==8||a.C[i]==16){if(n[i]){v=tX(Drb.NY,Qsb,6,16,15,1);v[0]=i;A=1;w=0;while(w<A){for(e=0;e<a.r;e++){b=-1;a.D[0][e]==v[w]?b=a.D[1][e]:a.D[1][e]==v[w]&&(b=a.D[0][e]);if(b==-1)continue;if(a.s[b]!=0){l=false;for(p=0;p<A;p++){if(b==v[p]){l=true;break;}}if(!l){if(A==v.length){j=tX(Drb.NY,Qsb,6,2*v.length,15,1);Erb.Y7(v,0,j,0,v.length);v=j;}v[A++]=b;}}}++w;}m=0;for(q=0;q<A;q++){m+=Jrb.mk(a,v[q]);}if(m<0){t=-1;u=99;for(o=0;o<A;o++){if(Jrb.mk(a,v[o])<0){if(u>Jrb.wq(Jrb.Dk(a,v[o]))){u=Jrb.wq(Jrb.Dk(a,v[o]));t=v[o];}}}if(t!=-1){B-=a.s[t];a.s[t]=0;a.T=0;}}}else {B-=a.s[i];a.s[i]=0;a.T=0;}}}}B>0&&(B=Jrb.yq(a,B));return B;};Jrb.yq=function yq(a,b){var c,d,e,f;b>0&&(b=Jrb.Bq(a,b,9));b>0&&(b=Jrb.Bq(a,b,17));b>0&&(b=Jrb.Bq(a,b,35));b>0&&(b=Jrb.Bq(a,b,53));if(b>0){for(c=0;c<a.f;c++){if(a.s[c]>0){e=false;for(f=0;f<a.j[c];f++){d=a.i[c][f];if(a.s[d]==0&&Jrb.ql(a,d)&&Jrb.Wn(a,d)>0){a.s[d]=-1;a.T=0;--b;if(b==0)return 0;e=true;break;}}if(e)continue;}}}b>0&&(b=Jrb.zq(a,b,8,16));b>0&&(b=Jrb.zq(a,b,8,15));b>0&&(b=Jrb.zq(a,b,8,6));b>0&&(b=Jrb.zq(a,b,7,16));return b;};Jrb.zq=function zq(a,b,c,d){var e,f,g,h,i,j,k,l;for(e=0;e<a.f;e++){if(a.C[e]==c&&a.s[e]==0&&Jrb.Wn(a,e)>0){h=false;for(j=0;j<a.j[e];j++){if(a.n[e][j]==1){f=a.i[e][j];if(a.C[f]==d){l=false;i=false;for(k=0;k<a.j[f];k++){g=a.i[f][k];if(a.s[g]<0){i=true;break;}g!=e&&a.C[g]==8&&a.n[f][k]==2&&(l=true);}if(!i&&l){a.s[e]=-1;a.T=0;--b;if(b==0)return 0;h=true;}}}if(h)break;}}}return b;};Jrb.Aq=function Aq(a,b,c,d){var e,f,g,h,i,j,k,l;l=tX(Drb.NY,Qsb,6,a.g,15,1);i=tX(Drb.c3,itb,6,a.g,16,1);for(g=0;g<a.g;g++){l[g]=a.H[g];if(b[g]){i[g]=true;a.H[g]=1;a.T=0;}}for(e=0;e<a.f;e++){if(e!=d&&a.o[e]==0&&i[a.k[d][0]]){for(k=0;k<a.j[e];k++){h=a.k[e][k];a.H[h]=1;a.T=0;i[h]=false;}}}for(j=0;j<a.j[c];j++){h=a.k[c][j];a.H[h]=1;a.T=0;i[h]=false;}if(Jrb.de(new Jrb.qe(a),i,false)){a.s[c]=0;a.T=0;a.j[d]!=a.e[d]&&Jrb.bk(a,Jrb.Fn(a,d,a.e[d]-1));Jrb.ou(a,7);return true;}for(f=0;f<a.g;f++){Jrb.tm(a,f,l[f]);}return false;};Jrb.Bq=function Bq(a,b,c){var d;for(d=0;d<a.f;d++){if(a.C[d]==c&&a.s[d]==0&&a.j[d]==0){a.s[d]=-1;a.T=0;--b;if(b==0)return 0;}}return b;};Jrb.Cq=function Cq(a,b){var c,d;d=isb.Lab(a.b,b);for(c=Hrb.rnb(d).length;c<10;c++)Erb.L7(a.a,32);Erb.P7(a.a,d);};Jrb.Dq=function Dq(a,b){var c,d,e;if(b<0||b>999){Erb.P7(a.a,'  ?');return;}c=false;for(d=0;d<3;d++){e=b/100|0;if(e==0){d==2||c?Erb.L7(a.a,48):Erb.L7(a.a,32);}else {Erb.L7(a.a,48+e&Gsb);c=true;}b=10*(b%100);}};Jrb.Eq=function Eq(a,b){fsb.A4(b,a.a.a);};Jrb.Fq=function Fq(a){Jrb.Gq.call(this,a);};Jrb.Gq=function Gq(a){Jrb.Iq.call(this,a,new Erb.T7());};Jrb.Hq=function Hq(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z;this.b=new isb.Oab((N=Yub,N));Jrb.ou(a,15);L=true;for(e=0;e<a.f;e++){if((a.u[e]&3)!=0&&(a.u[e]&3)!=3&&(a.u[e]&_tb)>>18!=1){L=false;break;}}O=-1;if(L){F=tX(Drb.NY,Qsb,6,32,15,1);P=0;for(f=0;f<a.f;f++){if((a.u[f]&3)!=0&&(a.u[f]&3)!=3&&(a.u[f]&_tb)>>18==1){H=(a.u[f]&_tb)>>18!=1&&(a.u[f]&_tb)>>18!=2?-1:(a.u[f]&Cub)>>20;++F[H];if(P<F[H]){P=F[H];O=H;}break;}}}this.a=!c?new Erb.S7():c;R=a.P!=null?a.P:'';Erb.P7(this.a,R+rsb);Erb.P7(this.a,'Actelion Java MolfileCreator 1.0\n\n');Jrb.Dq(this,a.q);Jrb.Dq(this,a.r);Erb.P7(this.a,'  0  0');Jrb.Dq(this,L?0:1);Erb.P7(this.a,'  0  0  0  0  0999 V2000\n');I=a.q==1;for(i=1;i<a.q;i++){if(a.J[i].a!=a.J[0].a||a.J[i].b!=a.J[0].b||a.J[i].c!=a.J[0].c){I=true;break;}}G=1;if(I){if(b!=0){G=b;}else {s=Jrb.Fk(a,a.q,a.r,(Jrb.Ej(),Jrb.Dj));if(s!=0){(s<1||s>3)&&(G=1.5/s);}else {Q=Eub;for(f=1;f<a.q;f++){for(h=0;h<f;h++){B=a.J[h].a-a.J[f].a;C=a.J[h].b-a.J[f].b;D=a.J[h].c-a.J[f].c;A=B*B+C*C+D*D;Q>A&&(Q=A);}}G=3/Q;}}}for(j=0;j<a.q;j++){if(I){Jrb.Cq(this,G*a.J[j].a);Jrb.Cq(this,G*-a.J[j].b);Jrb.Cq(this,G*-a.J[j].c);}else {Erb.P7(this.a,'    0.0000    0.0000    0.0000');}if((a.v==null?null:a.v[j])!=null)Erb.P7(this.a,' L  ');else if(E3(j3(a.B[j],1),0))Erb.P7(this.a,' A  ');else if(a.C[j]>=129&&a.C[j]<=144||a.C[j]==154)Erb.P7(this.a,' R# ');else {p=(Jrb.Ej(),Jrb.yj)[a.C[j]];Erb.P7(this.a,' '+p);Hrb.rnb(p).length==1?Erb.P7(this.a,'  '):Hrb.rnb(p).length==2&&Erb.P7(this.a,' ');}Erb.P7(this.a,' 0  0  0');J=j3(utb,a.B[j]);o3(J,0)==0?Erb.P7(this.a,'  0'):o3(J,384)==0?Erb.P7(this.a,'  3'):o3(J,128)==0?Erb.P7(this.a,'  2'):o3(J,1792)==0?Erb.P7(this.a,'  1'):o3(J,1664)==0&&Erb.P7(this.a,'  2');Erb.P7(this.a,E3(j3(a.B[j],Utb),0)?'  1':'  0');Z=((a.u[j]&Bub)>>>27)-1;Z==-1?Erb.P7(this.a,'  0'):Z==0?Erb.P7(this.a,' 15'):Jrb.Dq(this,Z);Erb.P7(this.a,'  0  0  0');Jrb.Dq(this,$wnd.Math.abs(a.w[j]));Erb.P7(this.a,'  0  0\n');}for(t=0;t<a.r;t++){switch(a.H[t]){case 1:T=1;W=0;break;case 2:T=2;W=0;break;case 4:T=3;W=0;break;case 129:T=1;W=6;break;case 257:T=1;W=1;break;case 386:T=2;W=3;break;case 8:T=4;W=0;break;case 16:T=8;W=0;break;default:T=1;W=0;}if(L&&(W==1||W==6)){f=a.D[0][t];(a.u[f]&_tb)>>18==2?W=0:(a.u[f]&_tb)>>18==1&&((a.u[f]&_tb)>>18!=1&&(a.u[f]&_tb)>>18!=2?-1:(a.u[f]&Cub)>>20)!=O&&(W=4);}u=a.G[t]&31;u!=0&&(u==8?T=4:u==3?T=5:u==9?T=6:u==10?T=7:T=8);V=a.G[t]&384;Y=V==0?0:V==256?1:2;Jrb.Dq(this,1+a.D[0][t]);Jrb.Dq(this,1+a.D[1][t]);Jrb.Dq(this,T);Jrb.Dq(this,W);Erb.P7(this.a,'  0');Jrb.Dq(this,Y);Erb.P7(this.a,'  0\n');}S=0;for(k=0;k<a.q;k++)a.s[k]!=0&&++S;if(S!=0){w=0;for(f=0;f<a.q;f++){if(a.s[f]!=0){if(w==0){Erb.P7(this.a,'M  CHG');Jrb.Dq(this,$wnd.Math.min(8,S));}Erb.P7(this.a,' ');Jrb.Dq(this,f+1);v=a.s[f];if(v<0){Erb.P7(this.a,'  -');v=-v;}else Erb.P7(this.a,'   ');Erb.L7(this.a,48+v&Gsb);--S;if(++w==8||S==0){w=0;Erb.P7(this.a,rsb);}}}}S=0;for(l=0;l<a.q;l++)a.A[l]==0||++S;if(S!=0){w=0;for(f=0;f<a.q;f++){if(a.A[f]!=0){if(w==0){Erb.P7(this.a,'M  ISO');Jrb.Dq(this,$wnd.Math.min(8,S));}Erb.P7(this.a,' ');Jrb.Dq(this,f+1);Erb.P7(this.a,' ');Jrb.Dq(this,a.A[f]);--S;if(++w==8||S==0){w=0;Erb.P7(this.a,rsb);}}}}S=0;for(m=0;m<a.q;m++)(a.u[m]&48)!=0&&++S;if(S!=0){w=0;for(f=0;f<a.q;f++){if((a.u[f]&48)!=0){if(w==0){Erb.P7(this.a,'M  RAD');Jrb.Dq(this,$wnd.Math.min(8,S));}Erb.P7(this.a,' ');Jrb.Dq(this,f+1);switch(a.u[f]&48){case 16:Erb.P7(this.a,'   1');break;case 32:Erb.P7(this.a,'   2');break;case 48:Erb.P7(this.a,'   3');}--S;if(++w==8||S==0){w=0;Erb.P7(this.a,rsb);}}}}S=0;for(n=0;n<a.q;n++)(a.C[n]>=129&&a.C[n]<=144||a.C[n]==154)&&++S;if(S!=0){w=0;for(d=0;d<a.q;d++){r=a.C[d];if(r>=129&&r<=144||r==154){if(w==0){Erb.P7(this.a,'M  RGP');Jrb.Dq(this,$wnd.Math.min(8,S));}Erb.P7(this.a,' ');Jrb.Dq(this,d+1);Erb.P7(this.a,' ');Jrb.Dq(this,r==154?0:r>=142?r-141:r-125);--S;if(++w==8||S==0){w=0;Erb.P7(this.a,rsb);}}}}if(a.K){S=0;for(f=0;f<a.q;f++)E3(j3(a.B[f],120),0)&&++S;if(S!=0){w=0;for(h=0;h<a.q;h++){U=j3(a.B[h],120);if(o3(U,0)!=0){if(w==0){Erb.P7(this.a,'M  RBC');Jrb.Dq(this,$wnd.Math.min(8,S));}Erb.P7(this.a,' ');Jrb.Dq(this,h+1);o3(U,112)==0?Erb.P7(this.a,'  -1'):o3(U,8)==0?Erb.P7(this.a,'   1'):o3(U,104)==0?Erb.P7(this.a,'   2'):o3(U,88)==0?Erb.P7(this.a,'   3'):o3(U,56)==0&&Erb.P7(this.a,'   4');--S;if(++w==8||S==0){w=0;Erb.P7(this.a,rsb);}}}}for(o=0;o<a.q;o++){q=a.v==null?null:a.v[o];if(q!=null){Erb.P7(this.a,'M  ALS ');Jrb.Dq(this,o+1);Jrb.Dq(this,q.length);Erb.P7(this.a,E3(j3(a.B[o],1),0)?' T ':' F ');for(K=0;K<q.length;K++){M=(Jrb.Ej(),Jrb.yj)[q[K]];switch(Hrb.rnb(M).length){case 1:Erb.P7(this.a,M+'   ');break;case 2:Erb.P7(this.a,M+'  ');break;case 3:Erb.P7(this.a,M+' ');break;default:Erb.P7(this.a,'   ?');}}Erb.P7(this.a,rsb);}}S=0;for(g=0;g<a.q;g++)E3(j3(a.B[g],6144),0)&&++S;if(S!=0){w=0;for(d=0;d<a.q;d++){X=j3(a.B[d],6144);if(o3(X,0)!=0){if(w==0){Erb.P7(this.a,'M  SUB');Jrb.Dq(this,$wnd.Math.min(8,S));}Erb.P7(this.a,' ');Jrb.Dq(this,d+1);E3(j3(X,qtb),0)?Erb.P7(this.a,'   '+(a.e[d]+1)):Erb.P7(this.a,'  -2');--S;if(++w==8||S==0){w=0;Erb.P7(this.a,rsb);}}}}}Erb.P7(this.a,Zub);};Jrb.Iq=function Iq(a,b){Jrb.Hq.call(this,a,0,b);};Z3(99,1,{},Jrb.Fq,Jrb.Hq);Drb.rZ=B5(99);Jrb.Jq=function Jq(a,b,c,d,e,f){var g,h,i,j;j=1;h=false;switch(e){case 1:j=257;break;case 3:j=386;break;case 4:j=257;h=true;break;case 6:j=129;break;default:switch(d){case 1:j=1;break;case 2:j=2;break;case 3:j=4;break;case 4:j=8;break;case 9:j=16;}}g=Jrb.Jj(a.j,b,c,j);i=0;h&&Jrb.Xl(a.j,b,1,-1);if(d>4){switch(d){case 5:i|=3;break;case 6:i|=9;break;case 7:i|=10;break;case 8:j!=16&&(i|=31);}}f==1&&(i|=256);f==2&&(i|=128);i!=0&&Jrb.sm(a.j,g,i,true);return g;};Jrb.Kq=function Kq(a){var b,c,d,e,f,g,h,i,j,k,l;g=tX(Drb.NY,Qsb,6,a.j.q,15,1);for(c=0;c<a.j.q;c++)g[c]=-Jrb.mk(a.j,c);Jrb.de(new Jrb.qe(a.j),null,true);for(d=0;d<a.j.q;d++)g[d]+=Jrb.mk(a.j,d);for(b=0;b<a.j.q;b++){if(g[b]!=0){h=-g[b];for(e=0;e<a.j.r;e++){for(j=0;j<2;j++){if(h>0&&Jrb.Tk(a.j,e)==16&&Jrb.Ik(a.j,1-j,e)==b){l=Jrb.Ik(a.j,j,e);if(Jrb.tl(a.j,l)){k=Jrb.Mq(a,l);f=Jrb.mk(a.j,l);if(f<k){i=$wnd.Math.min(h,k-f);Jrb.Sl(a.j,l,f+i);h-=i;}}}}}}}};Jrb.Lq=function Lq(a,b){a.j=null;return Jrb.br(a,new fsb.v4(new fsb.z4(b)))?a.j:null;};Jrb.Mq=function Mq(a,b){var c,d;c=Jrb.Dk(a.j,b);d=c<(Jrb.Ej(),Jrb.Aj).length?Jrb.Aj[c]:null;return d==null?0:d[d.length-1];};Jrb.Nq=function Nq(a,b){var c;c=!a.b?null:Krb.kdb(a.b,new Erb.o6(b));return !c?b-1:c.a;};Jrb.Oq=function Oq(a,b){var c;c=!a.c?null:Krb.kdb(a.c,new Erb.o6(b));return !c?b-1:c.a;};Jrb.Pq=function Pq(a,b){var c,d;if(b!=null){Jrb.ou(a.j,1);for(c=0;c<a.j.f;c++){if(b[c]!=0){d=Jrb.Xk(a.j,c,Jrb.ao(a.j,c));b[c]==15?d>=0&&Jrb.Ql(a.j,c,0):b[c]!=Jrb.Zk(a.j,c)&&Jrb.Ql(a.j,c,b[c]-d);}}}};Jrb.Qq=function Qq(a,b){var c;if(b==-1){return  -1;}for(c=b+1;c<Hrb.rnb(a).length;c++){Hrb.mnb(c,Hrb.rnb(a).length);if(Hrb.rnb(a).charCodeAt(c)!=32&&(Hrb.mnb(c,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(c)!=9)){return c;}}return  -1;};Jrb.Rq=function Rq(a,b){var c;for(c=b;c<Hrb.rnb(a).length;c++){Hrb.mnb(c,Hrb.rnb(a).length);if(Hrb.rnb(a).charCodeAt(c)==32||(Hrb.mnb(c,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(c)==9)){return c;}}return  -1;};Jrb.Sq=function Sq(a){if(Hrb.rnb(a).indexOf('ATOMS=(')!=-1)return $ub;if(Hrb.rnb(a).indexOf('BONDS=(')!=-1)return 'BONDS';return null;};Jrb.Tq=function Tq(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F;p=0;q=(h=Jrb.Rq(b,p+1),h==-1?Hrb.rnb(b).length:h);d=Erb.V5((Hrb.lnb(p,q,Hrb.rnb(b).length),Hrb.rnb(b).substr(p,q-p)),_ub,xsb);p=Jrb.Qq(b,q);q=(i=Jrb.Rq(b,p+1),i==-1?Hrb.rnb(b).length:i);s=(Hrb.lnb(p,q,Hrb.rnb(b).length),Hrb.rnb(b).substr(p,q-p));A=null;e=false;r=Jrb.$q(b);if(r!=0){A=Jrb.Uq(b);r<0&&(e=true);q=$wnd.Math.abs(r);}p=Jrb.Qq(b,q);q=(j=Jrb.Rq(b,p+1),j==-1?Hrb.rnb(b).length:j);C=g6((Hrb.lnb(p,q,Hrb.rnb(b).length),Hrb.rnb(b).substr(p,q-p)));p=Jrb.Qq(b,q);q=(k=Jrb.Rq(b,p+1),k==-1?Hrb.rnb(b).length:k);D=g6((Hrb.lnb(p,q,Hrb.rnb(b).length),Hrb.rnb(b).substr(p,q-p)));p=Jrb.Qq(b,q);q=(l=Jrb.Rq(b,p+1),l==-1?Hrb.rnb(b).length:l);F=g6((Hrb.lnb(p,q,Hrb.rnb(b).length),Hrb.rnb(b).substr(p,q-p)));p=Jrb.Qq(b,q);q=(m=Jrb.Rq(b,p+1),m==-1?Hrb.rnb(b).length:m);u=Erb.V5((Hrb.lnb(p,q,Hrb.rnb(b).length),Hrb.rnb(b).substr(p,q-p)),_ub,xsb);c=Jrb.Gj(a.j,C,-D,-F);c+1!=d&&(!a.b&&(a.b=new Krb.xjb()),Krb.pjb(a.b,new Erb.o6(d),new Erb.o6(c)));if(A!=null){Jrb.jm(a.j,c,6);Jrb.Zl(a.j,c,A,e);}u!=0&&Jrb.$l(a.j,c,u,false);if(Erb.k7(s,'A')||Erb.k7(s,'*')){Jrb.cm(a.j,c,1,true);}else if(Erb.k7(s,'Q')){t=tX(Drb.NY,Qsb,6,1,15,1);t[0]=6;Jrb.Zl(a.j,c,t,true);}else {Jrb.jm(a.j,c,Jrb.Xm(s,67));}while((p=Jrb.Qq(b,q))!=-1){q=(g=Jrb.Rq(b,p+1),g==-1?Hrb.rnb(b).length:g);v=(Hrb.lnb(p,q,Hrb.rnb(b).length),Hrb.rnb(b).substr(p,q-p));o=Erb.o7(v,A7(61));n=(Hrb.lnb(0,o,Hrb.rnb(v).length),Hrb.rnb(v).substr(0,o));B=Erb.V5((Hrb.mnb(o+1,Hrb.rnb(v).length+1),Hrb.rnb(v).substr(o+1)),_ub,xsb);if(Erb.k7(n,'CHG')){Jrb.Sl(a.j,c,B);}else if(Erb.k7(n,'RAD')){switch(B){case 1:Jrb.dm(a.j,c,16);break;case 2:Jrb.dm(a.j,c,32);break;case 3:Jrb.dm(a.j,c,48);}}else if(Erb.k7(n,'CFG'));else if(Erb.k7(n,'MASS')){Jrb.am(a.j,c,B);}else if(Erb.k7(n,'VAL')){Jrb.Ql(a.j,c,B==-1?0:B==0?-1:B);}else if(Erb.k7(n,'HCOUNT')){switch(B){case 0:break;case  -1:Jrb.cm(a.j,c,1792,true);break;case 1:Jrb.cm(a.j,c,128,true);break;case 2:Jrb.cm(a.j,c,384,true);break;default:Jrb.cm(a.j,c,896,true);}}else if(Erb.k7(n,'SUBST')){if(B==-1){Jrb.cm(a.j,c,Vtb,true);}else if(B>0){w=0;for(f=0;f<a.j.r;f++){(Jrb.Ik(a.j,0,f)==c||Jrb.Ik(a.j,1,f)==c)&&++w;}B>w&&Jrb.cm(a.j,c,qtb,true);}}else if(Erb.k7(n,'RBCNT')){switch(B){case 3:case  -1:Jrb.cm(a.j,c,112,true);break;case 1:Jrb.cm(a.j,c,8,true);break;case 2:Jrb.cm(a.j,c,104,true);break;case 4:Jrb.cm(a.j,c,56,true);}}}};Jrb.Uq=function Uq(a){var b,c,d,e,f,g,h,i;h=null;c=Hrb.rnb(a).indexOf('[');d=Hrb.rnb(a).indexOf(']',c);if(c>=0&&d>0){b=tX(Drb.NY,Qsb,6,16,15,1);i=(Hrb.lnb(c+1,d,Hrb.rnb(a).length),Hrb.rnb(a).substr(c+1,d-(c+1)));e=0;g=true;while(g&&e<16){c=Hrb.rnb(i).indexOf(',');f=null;if(c==-1){f=i;g=false;}else {f=(Hrb.lnb(0,c,Hrb.rnb(i).length),Hrb.rnb(i).substr(0,c));i=(Hrb.mnb(c+1,Hrb.rnb(i).length+1),Hrb.rnb(i).substr(c+1));}b[e++]=Jrb.Xm(f,1);}h=tX(Drb.NY,Qsb,6,e,15,1);Erb.Y7(b,0,h,0,e);}return h;};Jrb.Vq=function Vq(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t;o=0;p=(i=Jrb.Rq(b,o+1),i==-1?Hrb.rnb(b).length:i);f=Erb.V5((Hrb.lnb(o,p,Hrb.rnb(b).length),Hrb.rnb(b).substr(o,p-o)),_ub,xsb);o=Jrb.Qq(b,p);p=(j=Jrb.Rq(b,o+1),j==-1?Hrb.rnb(b).length:j);g=Erb.V5((Hrb.lnb(o,p,Hrb.rnb(b).length),Hrb.rnb(b).substr(o,p-o)),_ub,xsb);o=Jrb.Qq(b,p);p=(k=Jrb.Rq(b,o+1),k==-1?Hrb.rnb(b).length:k);c=Jrb.Nq(a,Erb.V5((Hrb.lnb(o,p,Hrb.rnb(b).length),Hrb.rnb(b).substr(o,p-o)),_ub,xsb));o=Jrb.Qq(b,p);p=(l=Jrb.Rq(b,o+1),l==-1?Hrb.rnb(b).length:l);d=Jrb.Nq(a,Erb.V5((Hrb.lnb(o,p,Hrb.rnb(b).length),Hrb.rnb(b).substr(o,p-o)),_ub,xsb));r=0;s=0;while((o=Jrb.Qq(b,p))!=-1){p=(h=Jrb.Rq(b,o+1),h==-1?Hrb.rnb(b).length:h);q=(Hrb.lnb(o,p,Hrb.rnb(b).length),Hrb.rnb(b).substr(o,p-o));n=Erb.o7(q,A7(61));m=(Hrb.lnb(0,n,Hrb.rnb(q).length),Hrb.rnb(q).substr(0,n));t=Erb.V5((Hrb.mnb(n+1,Hrb.rnb(q).length+1),Hrb.rnb(q).substr(n+1)),_ub,xsb);if(Erb.k7(m,'CFG')){switch(t){case 1:r=1;break;case 2:r=g==2?3:4;break;case 3:r=6;}}else Erb.k7(m,'TOPO')?s=t:undefined;}e=Jrb.Jq(a,c,d,g,r,s);e+1!=f&&(!a.c&&(a.c=new Krb.xjb()),Krb.pjb(a.c,new Erb.o6(f),new Erb.o6(e)));};Jrb.Wq=function Wq(a,b){var c,d,e,f,g,h;h=Jrb.Sq(b);if(h!=null){g=Jrb.Yq(b,h);if(Erb.k7(Hrb.rnb(b).substr(0,13),'MDLV30/STEABS')){if(Erb.k7(h,$ub))for(f=0;f<g.length;f++)Jrb.Xl(a.j,Jrb.Nq(a,g[f]),0,-1);else for(e=0;e<g.length;e++)Jrb.nm(a.j,Jrb.Oq(a,g[e]),0,-1);}else if(Erb.k7(Hrb.rnb(b).substr(0,13),'MDLV30/STERAC')){d=Erb.V5(Erb.x7(b,13,Jrb.Rq(b,13)),_ub,xsb);if(Erb.k7(h,$ub))for(f=0;f<g.length;f++)Jrb.Xl(a.j,Jrb.Nq(a,g[f]),1,d-1);else for(e=0;e<g.length;e++)Jrb.nm(a.j,Jrb.Oq(a,g[e]),1,d-1);}else if(Erb.k7(Hrb.rnb(b).substr(0,13),'MDLV30/STEREL')){d=Erb.V5(Erb.x7(b,13,Jrb.Rq(b,13)),_ub,xsb);if(Erb.k7(h,$ub))for(f=0;f<g.length;f++)Jrb.Xl(a.j,Jrb.Nq(a,g[f]),2,d-1);else for(e=0;e<g.length;e++)Jrb.nm(a.j,Jrb.Oq(a,g[e]),2,d-1);}else if(Erb.k7(Hrb.rnb(b).substr(0,13),'MDLV30/HILITE')){if(Erb.k7(h,$ub)){for(e=0;e<g.length;e++)Jrb.Tl(a.j,Jrb.Nq(a,g[e]),448);}else {for(e=0;e<g.length;e++){c=Jrb.Oq(a,g[e]);Jrb.Tl(a.j,Jrb.Ik(a.j,0,c),448);Jrb.Tl(a.j,Jrb.Ik(a.j,1,c),448);}}}}};Jrb.Xq=function Xq(a,b){var c,d,e;if(!a.j){if(Erb.k7(Hrb.rnb(b).substr(0,6),'COUNTS')){c=Jrb.Qq(b,Jrb.Rq(b,7));d=Erb.V5(Erb.x7(b,7,Jrb.Rq(b,7)),_ub,xsb);e=Erb.V5(Erb.x7(b,c,Jrb.Rq(b,c)),_ub,xsb);a.j=new Jrb.Hu(d,e);}}};Jrb.Yq=function Yq(a,b){var c,d,e,f,g,h;f=Hrb.rnb(a).indexOf(b+'=(')+Hrb.rnb(b).length+2;g=Erb.p7(a,A7(41),f);e=Jrb.Rq(a,f);c=Erb.V5((Hrb.lnb(f,e,Hrb.rnb(a).length),Hrb.rnb(a).substr(f,e-f)),_ub,xsb);h=tX(Drb.NY,Qsb,6,c,15,1);for(d=0;d<c;d++){f=Jrb.Qq(a,e);e=Jrb.Rq(a,f);(e==-1||e>g)&&(e=g);h[d]=Erb.V5((Hrb.lnb(f,e,Hrb.rnb(a).length),Hrb.rnb(a).substr(f,e-f)),_ub,xsb);}return h;};Jrb.Zq=function Zq(a){var b,c,d,e,f,g,h,i,j;i=tX(Drb.NY,Qsb,6,a.j.q,15,1);for(d=0;d<a.j.r;d++)if(Jrb.Tk(a.j,d)==8)for(g=0;g<2;g++)i[Jrb.Ik(a.j,g,d)]=1;for(e=0;e<a.j.r;e++){j=Jrb.Qk(a.j,e);for(f=0;f<2;f++)i[Jrb.Ik(a.j,f,e)]+=j;}for(c=0;c<a.j.r;c++){if(Jrb.Qk(a.j,c)==1){for(f=0;f<2;f++){h=Jrb.Ik(a.j,1-f,c);if(Jrb.tl(a.j,h)){b=Jrb.Ik(a.j,f,c);if(Jrb.ql(a.j,b)&&i[b]>Jrb.Zk(a.j,b)){Jrb.tm(a.j,c,16);continue;}}}}}};Jrb.$q=function $q(a){var b,c;if(Hrb.rnb(a).indexOf('[')>=0){b=Hrb.rnb(a).indexOf(' NOT[');c=Hrb.rnb(a).indexOf(']',b);if(b>=0&&c>0){return -(c+1);}else {b=Hrb.rnb(a).indexOf(' [');c=Hrb.rnb(a).indexOf(']',b);if(b>=0&&c>0){return c+1;}}b=Hrb.rnb(a).indexOf(" 'NOT[");c=Hrb.rnb(a).indexOf("]'",b);if(b>=0&&c>0){return -(c+2);}else {b=Hrb.rnb(a).indexOf(" '[");c=Hrb.rnb(a).indexOf("]'",b);if(b>=0&&c>0){return c+2;}}Erb.X7();}return 0;};Jrb._q=function _q(a,b,c){a.j=b;return Jrb.br(a,c);};Jrb.ar=function ar(a){return Hrb.rnb(a).length==0?0:Erb.V5(a,_ub,xsb);};Jrb.br=function br(b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X;T=null;try{b.f=null;if(b.j){Jrb.mu(b.j);Jrb.vm(b.j,false);}D=v=fsb.u4(c);if(null==D){return false;}if(null==(v=fsb.u4(c))){return false;}if(null==(v=fsb.u4(c))){return false;}b.k=Hrb.rnb(v).indexOf("From CSD data. Using bond type 'Any'")!=-1;b.e=Hrb.rnb(v).indexOf('From CSD data.')!=-1;if(null==(v=fsb.u4(c))){return false;}b.g=false;b.d=b.a;try{F=Erb.V5(Erb.y7((Hrb.lnb(0,3,Hrb.rnb(v).length),Hrb.rnb(v).substr(0,3))),_ub,xsb);G=Erb.V5(Erb.y7((Hrb.lnb(3,6,Hrb.rnb(v).length),Hrb.rnb(v).substr(3,3))),_ub,xsb);H=Jrb.ar(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))));b.d=b.d|1==Jrb.ar(Erb.y7((Hrb.lnb(12,15,Hrb.rnb(v).length),Hrb.rnb(v).substr(12,3))));b.g=Hrb.rnb(v).length>=39&&Erb.k7(Hrb.rnb(v).substr(34,Hrb.rnb('V3000').length),'V3000');}catch(a){a=g3(a);if(zY(a,15)){return false;}else throw h3(a);}if(b.g){K=Jrb.cr(b,c);Jrb.Bm(b.j,D);return K;}!b.j&&(b.j=new Jrb.Hu(F,G));Jrb.Bm(b.j,D);b.d||(b.j.L=true);if(0==F){while(v!=null&&!(Erb.k7(v,avb)||Erb.k7(v,bvb)||Erb.k7((Hrb.mnb(1,Hrb.rnb(v).length+1),Hrb.rnb(v).substr(1)),'$'))){v=fsb.u4(c);}return true;}for(q=0;q<F;q++){if(null==(v=fsb.u4(c))){return false;}V=g6(Erb.y7((Hrb.lnb(0,10,Hrb.rnb(v).length),Hrb.rnb(v).substr(0,10))));W=g6(Erb.y7((Hrb.lnb(10,20,Hrb.rnb(v).length),Hrb.rnb(v).substr(10,10))));X=g6(Erb.y7((Hrb.lnb(20,30,Hrb.rnb(v).length),Hrb.rnb(v).substr(20,10))));e=Jrb.Gj(b.j,V,-W,-X);u=Erb.y7((Hrb.lnb(31,34,Hrb.rnb(v).length),Hrb.rnb(v).substr(31,3)));if(Erb.k7(u,'A')||Erb.k7(u,'*')){Jrb.cm(b.j,e,1,true);}else if(Erb.k7(u,'Q')){w=tX(Drb.NY,Qsb,6,1,15,1);w[0]=6;Jrb.Zl(b.j,e,w,true);}else {h=Jrb.Xm(u,67);Jrb.jm(b.j,e,h);}C=Jrb.ar(Erb.y7((Hrb.lnb(34,36,Hrb.rnb(v).length),Hrb.rnb(v).substr(34,2))));C!=0&&Jrb.am(b.j,e,(Jrb.Ej(),Jrb.Cj)[Jrb.Dk(b.j,e)]+C);m=Jrb.ar(Erb.y7((Hrb.lnb(36,39,Hrb.rnb(v).length),Hrb.rnb(v).substr(36,3))));m!=0&&(m==4?Jrb.dm(b.j,e,32):Jrb.Sl(b.j,e,4-m));A=Hrb.rnb(v).length<63?0:Jrb.ar(Erb.y7((Hrb.lnb(60,63,Hrb.rnb(v).length),Hrb.rnb(v).substr(60,3))));Jrb.$l(b.j,e,A,false);o=Hrb.rnb(v).length<45?0:Jrb.ar(Erb.y7((Hrb.lnb(42,45,Hrb.rnb(v).length),Hrb.rnb(v).substr(42,3))));switch(o){case 0:break;case 1:Jrb.cm(b.j,e,768,true);break;case 2:Jrb.cm(b.j,e,128,true);break;case 3:Jrb.cm(b.j,e,384,true);break;default:Jrb.cm(b.j,e,896,true);}Hrb.rnb(v).length>=48&&(Hrb.mnb(47,Hrb.rnb(v).length),Hrb.rnb(v).charCodeAt(47)==49)&&Jrb.cm(b.j,e,Utb,true);S=Hrb.rnb(v).length<51?0:Jrb.ar(Erb.y7((Hrb.lnb(48,51,Hrb.rnb(v).length),Hrb.rnb(v).substr(48,3))));if(S!=0){T==null&&(T=tX(Drb.NY,Qsb,6,F,15,1));T[e]=S;}}for(r=0;r<G;r++){if(null==(v=fsb.u4(c))){return false;}f=Erb.V5(Erb.y7((Hrb.lnb(0,3,Hrb.rnb(v).length),Hrb.rnb(v).substr(0,3))),_ub,xsb)-1;g=Erb.V5(Erb.y7((Hrb.lnb(3,6,Hrb.rnb(v).length),Hrb.rnb(v).substr(3,3))),_ub,xsb)-1;k=Erb.V5(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))),_ub,xsb);N=Hrb.rnb(v).length<12?0:Jrb.ar(Erb.y7((Hrb.lnb(9,12,Hrb.rnb(v).length),Hrb.rnb(v).substr(9,3))));R=Hrb.rnb(v).length<18?0:Jrb.ar(Erb.y7((Hrb.lnb(15,18,Hrb.rnb(v).length),Hrb.rnb(v).substr(15,3))));k==8&&(b.k||Jrb.tl(b.j,f)||Jrb.tl(b.j,g))&&(k=9);Jrb.Jq(b,f,g,k,N,R);}for(p=0;p<H;p++){if(null==(v=fsb.u4(c))){return false;}}if(null==(v=fsb.u4(c))){(b.i&1)!=0&&(b.f=Jrb.Tn(b.j));Jrb.Pq(b,T);b.d||Jrb.ou(b.j,15);return true;}while(v!=null&&!(Erb.k7(v,avb)||Erb.k7(v,bvb))){if(Erb.k7(Hrb.rnb(v).substr(0,6),'M  CHG')){s=Erb.V5(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))),_ub,xsb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=Erb.V5(Erb.y7((Hrb.lnb(d,d+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(d,d+3-d))),_ub,xsb)-1;l=Erb.V5(Erb.y7((Hrb.lnb(U,U+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(U,U+3-U))),_ub,xsb);Jrb.Sl(b.j,e,l);}}}if(Erb.k7(Hrb.rnb(v).substr(0,6),'M  ISO')){s=Erb.V5(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))),_ub,xsb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=Erb.V5(Erb.y7((Hrb.lnb(d,d+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(d,d+3-d))),_ub,xsb)-1;B=Erb.V5(Erb.y7((Hrb.lnb(U,U+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(U,U+3-U))),_ub,xsb);Jrb.am(b.j,e,B);}}}if(Erb.k7(Hrb.rnb(v).substr(0,6),'M  RAD')){s=Erb.V5(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))),_ub,xsb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=Erb.V5(Erb.y7((Hrb.lnb(d,d+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(d,d+3-d))),_ub,xsb)-1;J=Erb.V5(Erb.y7((Hrb.lnb(U,U+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(U,U+3-U))),_ub,xsb);switch(J){case 1:Jrb.dm(b.j,e,16);break;case 2:Jrb.dm(b.j,e,32);break;case 3:Jrb.dm(b.j,e,48);}}}}if(Erb.k7(Hrb.rnb(v).substr(0,6),'M  RBC')||Erb.k7(Hrb.rnb(v).substr(0,6),'M  RBD')){s=Erb.V5(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))),_ub,xsb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=Erb.V5(Erb.y7((Hrb.lnb(d,d+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(d,d+3-d))),_ub,xsb)-1;L=Erb.V5(Erb.y7((Hrb.lnb(U,U+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(U,U+3-U))),_ub,xsb);switch(L){case 3:case -1:Jrb.cm(b.j,e,112,true);break;case 1:Jrb.cm(b.j,e,8,true);break;case 2:Jrb.cm(b.j,e,104,true);break;case 4:Jrb.cm(b.j,e,56,true);}}}}if(Erb.k7(Hrb.rnb(v).substr(0,6),'M  ALS')){e=Erb.V5(Erb.y7((Hrb.lnb(7,10,Hrb.rnb(v).length),Hrb.rnb(v).substr(7,3))),_ub,xsb)-1;if(e>=0){I=Erb.V5(Erb.y7((Hrb.lnb(10,13,Hrb.rnb(v).length),Hrb.rnb(v).substr(10,3))),_ub,xsb);i=(Hrb.mnb(14,Hrb.rnb(v).length),Hrb.rnb(v).charCodeAt(14)==84);S=tX(Drb.NY,Qsb,6,I,15,1);d=16;for(t=0;t<I;++t,d+=4){Q=Erb.y7((Hrb.lnb(d,d+4,Hrb.rnb(v).length),Hrb.rnb(v).substr(d,d+4-d)));S[t]=Jrb.Xm(Q,1);}Jrb.jm(b.j,e,6);Jrb.Zl(b.j,e,S,i);}}if(Erb.k7(Hrb.rnb(v).substr(0,6),'M  SUB')){s=Erb.V5(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))),_ub,xsb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=Erb.V5(Erb.y7((Hrb.lnb(d,d+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(d,d+3-d))),_ub,xsb)-1;O=Erb.V5(Erb.y7((Hrb.lnb(U,U+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(U,U+3-U))),_ub,xsb);if(O==-2){Jrb.cm(b.j,e,Vtb,true);}else if(O>0){P=0;for(j=0;j<b.j.r;j++){(Jrb.Ik(b.j,0,j)==e||Jrb.Ik(b.j,1,j)==e)&&++P;}O>P&&Jrb.cm(b.j,e,qtb,true);}}}}if(Erb.k7(Hrb.rnb(v).substr(0,6),'M  RGP')){s=Erb.V5(Erb.y7((Hrb.lnb(6,9,Hrb.rnb(v).length),Hrb.rnb(v).substr(6,3))),_ub,xsb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=Erb.V5(Erb.y7((Hrb.lnb(d,d+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(d,d+3-d))),_ub,xsb)-1;M=Erb.V5(Erb.y7((Hrb.lnb(U,U+3,Hrb.rnb(v).length),Hrb.rnb(v).substr(U,U+3-U))),_ub,xsb);M>=1&&M<=20&&Jrb.jm(b.j,e,Jrb.Xm('R'+M,2));}}}v=fsb.u4(c);}}catch(a){a=g3(a);if(zY(a,15)){n=a;Erb.Jz(n,(Erb.X7(),Erb.W7),'');return false;}else throw h3(a);}if(b.e){Jrb.Zq(b);Jrb.Kq(b);}(b.i&1)!=0&&(b.f=Jrb.Tn(b.j));Jrb.Pq(b,T);Jrb.ou(b.j,15);return true;};Jrb.cr=function cr(a,b){var c,d,e,f,g;!!a.b&&Krb.ijb(a.b);!!a.c&&Krb.ijb(a.c);e=0;d=fsb.u4(b);while(d!=null&&Erb.k7(Hrb.rnb(d).substr(0,7),cvb)){d=Erb.y7((Hrb.mnb(7,Hrb.rnb(d).length+1),Hrb.rnb(d).substr(7)));while(g=Hrb.rnb('-').length,Erb.k7(Hrb.rnb(d).substr(Hrb.rnb(d).length-g,g),'-')){c=fsb.u4(b);if(!Erb.k7(Hrb.rnb(c).substr(0,7),cvb)){return false;}d=Erb.y7(Erb.g7(Erb.x7(d,0,Hrb.rnb(d).length-1),(Hrb.mnb(7,Hrb.rnb(c).length+1),Hrb.rnb(c).substr(7))));}if(Erb.k7(Hrb.rnb(d).substr(0,5),'BEGIN')){f=Erb.y7((Hrb.mnb(6,Hrb.rnb(d).length+1),Hrb.rnb(d).substr(6)));if(Erb.k7(Hrb.rnb(f).substr(0,4),'CTAB')){e=1;}else if(Erb.k7(Hrb.rnb(f).substr(0,4),'ATOM')){e=2;}else if(Erb.k7(Hrb.rnb(f).substr(0,4),'BOND')){e=3;}else if(Erb.k7(Hrb.rnb(f).substr(0,10),'COLLECTION')){e=4;}else {return false;}}else if(Erb.k7(Hrb.rnb(d).substr(0,3),'END')){e=0;}else if(e==1){Jrb.Xq(a,d);}else if(e==2){Jrb.Tq(a,d);}else if(e==3){Jrb.Vq(a,d);}else if(e==4){Jrb.Wq(a,d);}else {return false;}d=fsb.u4(b);}while(d!=null&&!(Erb.k7(Hrb.rnb(d).substr(0,6),avb)||Erb.k7(d,bvb))){d=fsb.u4(b);}return true;};Jrb.dr=function dr(){this.i=0;};Jrb.er=function er(){this.i=1;};Z3(110,1,{},Jrb.dr,Jrb.er);_.a=false;_.d=false;_.e=false;_.g=false;_.i=0;_.k=false;Drb.sZ=B5(110);Jrb.fr=function fr(a){};Jrb.gr=function gr(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J;Erb.P7(a.a,'M  V30 BEGIN CTAB\n');Erb.P7(a.a,dvb+b.q+' '+b.r+' 0 0 0'+rsb);Erb.P7(a.a,'M  V30 BEGIN ATOM\n');for(i=0;i<b.q;i++){Erb.P7(a.a,cvb+(i+1));if((b.v==null?null:b.v[i])!=null){k=b.v==null?null:b.v[i];u=E3(j3(b.B[i],1),0);Erb.P7(a.a,u?' NOT[':' [');for(s=0;s<k.length;s++){s>0&&Erb.P7(a.a,',');t=(Jrb.Ej(),Jrb.yj)[k[s]];switch(Hrb.rnb(t).length){case 2:case 3:case 1:Erb.P7(a.a,t);break;default:Erb.P7(a.a,'?');}}Erb.P7(a.a,']');}else E3(j3(b.B[i],1),0)?Erb.P7(a.a,' A'):b.C[i]>=129&&b.C[i]<=144||b.C[i]==154?Erb.P7(a.a,' R#'):Erb.P7(a.a,' '+(Jrb.Ej(),Jrb.yj)[b.C[i]]);if(c){Erb.P7(a.a,' '+IY(Hsb*a.b*b.J[i].a)/Hsb);Erb.P7(a.a,' '+IY(Hsb*a.b*-b.J[i].b)/Hsb);Erb.P7(a.a,' '+IY(Hsb*a.b*-b.J[i].c)/Hsb);}else {Erb.P7(a.a,' 0 0 0');}Erb.P7(a.a,' '+$wnd.Math.abs(b.w[i]));b.s[i]!=0&&Erb.P7(a.a,' CHG='+b.s[i]);if((b.u[i]&48)!=0){Erb.P7(a.a,' RAD=');switch(b.u[i]&48){case 16:Erb.P7(a.a,'1');break;case 32:Erb.P7(a.a,'2');break;case 48:Erb.P7(a.a,'3');}}if((b.u[i]&3)==1||(b.u[i]&3)==2){Erb.P7(a.a,' CFG=');(b.u[i]&3)==1?Erb.P7(a.a,'1'):Erb.P7(a.a,'2');}b.A[i]!=0&&Erb.P7(a.a,' MASS='+b.A[i]);J=((b.u[i]&Bub)>>>27)-1;J!=-1&&Erb.P7(a.a,' VAL='+(J==0?'-1':Erb.A6(J)));l=b.C[i];(l>=129&&l<=144||l==154)&&Erb.P7(a.a,' RGROUPS=(1 '+(l==154?0:l>=142?l-141:l-125)+')');r=j3(utb,b.B[i]);o3(r,384)==0?Erb.P7(a.a,' HCOUNT=2'):o3(r,128)==0?Erb.P7(a.a,' HCOUNT=1'):o3(r,1792)==0?Erb.P7(a.a,' HCOUNT=-1'):o3(r,1664)==0&&Erb.P7(a.a,' HCOUNT=1');G=j3(b.B[i],6144);o3(G,0)!=0&&(E3(j3(G,qtb),0)?Erb.P7(a.a,' SUBST='+(b.e[i]+1)):Erb.P7(a.a,' SUBST=-1'));C=j3(b.B[i],120);o3(C,0)!=0&&(o3(C,112)==0?Erb.P7(a.a,' RBCNT=-1'):o3(C,8)==0?Erb.P7(a.a,' RBCNT=2'):o3(C,104)==0?Erb.P7(a.a,' RBCNT=2'):o3(C,88)==0?Erb.P7(a.a,' RBCNT=3'):o3(C,56)==0&&Erb.P7(a.a,' RBCNT=4'));Erb.P7(a.a,rsb);}Erb.P7(a.a,'M  V30 END ATOM\n');Erb.P7(a.a,'M  V30 BEGIN BOND\n');for(n=0;n<b.r;n++){Erb.P7(a.a,cvb+(n+1));switch(b.H[n]){case 1:A=1;F=0;break;case 2:A=2;F=0;break;case 4:A=3;F=0;break;case 129:A=1;F=3;break;case 257:A=1;F=1;break;case 386:A=2;F=2;break;case 8:A=4;F=0;break;case 16:A=9;F=0;break;default:A=1;F=0;}p=b.G[n]&31;p!=0&&(p==8?A=4:p==3?A=5:p==9?A=6:p==10?A=7:A=8);Erb.P7(a.a,' '+A+' '+(b.D[0][n]+1)+' '+(b.D[1][n]+1));F!=0&&Erb.P7(a.a,' CFG='+F);D=b.G[n]&384;H=D==0?0:D==256?1:2;H!=0&&Erb.P7(a.a,' TOPO='+H);Erb.P7(a.a,rsb);}Erb.P7(a.a,'M  V30 END BOND\n');B=false;d=0;v=tX(Drb.NY,Qsb,6,32,15,1);f=tX(Drb.NY,Qsb,6,32,15,1);for(j=0;j<b.f;j++){if((b.u[j]&3)==1||(b.u[j]&3)==2){B=true;I=(b.u[j]&_tb)>>18;I==1?++f[(b.u[j]&_tb)>>18!=1&&(b.u[j]&_tb)>>18!=2?-1:(b.u[j]&Cub)>>20]:I==2?++v[(b.u[j]&_tb)>>18!=1&&(b.u[j]&_tb)>>18!=2?-1:(b.u[j]&Cub)>>20]:++d;}}e=0;w=tX(Drb.NY,Qsb,6,32,15,1);g=tX(Drb.NY,Qsb,6,32,15,1);for(o=0;o<b.g;o++){if(Jrb.Qk(b,o)!=2&&((b.F[o]&3)==1||(b.F[o]&3)==2)){B=true;I=(b.F[o]&768)>>8;I==1?++g[(b.F[o]&768)>>8!=1&&(b.F[o]&768)>>8!=2?-1:(b.F[o]&Dub)>>10]:I==2?++w[(b.F[o]&768)>>8!=1&&(b.F[o]&768)>>8!=2?-1:(b.F[o]&Dub)>>10]:++e;}}if(B){Erb.P7(a.a,'M  V30 BEGIN COLLECTION\n');if(d!=0){Erb.P7(a.a,'M  V30 MDLV30/STEABS ATOMS=('+d);for(h=0;h<b.f;h++){((b.u[h]&3)==1||(b.u[h]&3)==2)&&(b.u[h]&_tb)>>18==0&&Erb.P7(a.a,' '+(h+1));}Erb.P7(a.a,evb);}if(e!=0){Erb.P7(a.a,'M  V30 MDLV30/STEABS BONDS=('+e);for(m=0;m<b.g;m++){Jrb.Qk(b,m)!=2&&((b.F[m]&3)==1||(b.F[m]&3)==2)&&(b.F[m]&768)>>8==0&&Erb.P7(a.a,' '+(m+1));}Erb.P7(a.a,evb);}for(q=0;q<32;q++){if(v[q]!=0){Erb.P7(a.a,fvb+(q+1)+' ATOMS=('+v[q]);for(h=0;h<b.f;h++){((b.u[h]&3)==1||(b.u[h]&3)==2)&&(b.u[h]&_tb)>>18==2&&((b.u[h]&_tb)>>18!=1&&(b.u[h]&_tb)>>18!=2?-1:(b.u[h]&Cub)>>20)==q&&Erb.P7(a.a,' '+(h+1));}Erb.P7(a.a,evb);}if(f[q]!=0){Erb.P7(a.a,gvb+(q+1)+' ATOMS=('+f[q]);for(h=0;h<b.f;h++){((b.u[h]&3)==1||(b.u[h]&3)==2)&&(b.u[h]&_tb)>>18==1&&((b.u[h]&_tb)>>18!=1&&(b.u[h]&_tb)>>18!=2?-1:(b.u[h]&Cub)>>20)==q&&Erb.P7(a.a,' '+(h+1));}Erb.P7(a.a,evb);}if(w[q]!=0){Erb.P7(a.a,fvb+(q+1)+' BONDS=('+w[q]);for(m=0;m<b.g;m++){Jrb.Qk(b,m)!=2&&((b.F[m]&3)==1||(b.F[m]&3)==2)&&(b.F[m]&768)>>8==2&&((b.F[m]&768)>>8!=1&&(b.F[m]&768)>>8!=2?-1:(b.F[m]&Dub)>>10)==q&&Erb.P7(a.a,' '+(m+1));}Erb.P7(a.a,evb);}if(g[q]!=0){Erb.P7(a.a,gvb+(q+1)+' BONDS=('+g[q]);for(m=0;m<b.g;m++){Jrb.Qk(b,m)!=2&&((b.F[m]&3)==1||(b.F[m]&3)==2)&&(b.F[m]&768)>>8==1&&((b.F[m]&768)>>8!=1&&(b.F[m]&768)>>8!=2?-1:(b.F[m]&Dub)>>10)==q&&Erb.P7(a.a,' '+(m+1));}Erb.P7(a.a,evb);}}Erb.P7(a.a,'M  V30 END COLLECTION\n');}Erb.P7(a.a,'M  V30 END CTAB\n');};Jrb.hr=function hr(){Jrb.fr(this);this.a=new Erb.T7();};Jrb.ir=function ir(a){Jrb.jr.call(this,a);};Jrb.jr=function jr(a){Jrb.kr.call(this,a,new Erb.T7());};Jrb.kr=function kr(a,b){var c,d;Jrb.fr(this);Jrb.ou(a,15);this.a=b;d=a.P!=null?a.P:'';Erb.P7(this.a,d+rsb);Erb.P7(this.a,'Actelion Java MolfileCreator 2.0\n\n');Erb.P7(this.a,'  0  0  0  0  0  0              0 V3000\n');this.b=1;c=Jrb.mr(a);c&&(this.b=Jrb.lr(a));Jrb.gr(this,a,c);Erb.P7(this.a,Zub);};Jrb.lr=function lr(a){var b,c,d,e,f,g,h,i,j;j=1;d=Jrb.Fk(a,a.q,a.r,(Jrb.Ej(),Jrb.Dj));if(d!=0){(d<1||d>3)&&(j=1.5/d);}else {i=hvb;for(b=1;b<a.q;b++){for(c=0;c<b;c++){f=a.J[c].a-a.J[b].a;g=a.J[c].b-a.J[b].b;h=a.J[c].c-a.J[b].c;e=f*f+g*g+h*h;i>e&&(i=e);}}j=3/$wnd.Math.max(0.75,i);}return j;};Jrb.mr=function mr(a){var b;if(a.q==1)return true;for(b=1;b<a.q;b++){if(a.J[b].a!=a.J[0].a||a.J[b].b!=a.J[0].b||a.J[b].c!=a.J[0].c){return true;}}return false;};Jrb.nr=function nr(a,b){var c;c=new Jrb.hr();c.b=b;Jrb.ou(a,15);Jrb.gr(c,a,true);return c.a.a;};Z3(125,1,{},Jrb.hr,Jrb.ir);_.b=1;Drb.tZ=B5(125);Jrb.or=function or(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;j=a.g.M;k=0;for(f=0;f<c;f++){if(j>b[f]){j=b[f];k=f;}}p=tX(Drb.NY,Qsb,6,c,15,1);i=k>0?k-1:c-1;l=k<c-1?k+1:0;g=b[i]<b[l];for(e=0;e<c;e++){p[e]=b[k];g?--k<0&&(k=c-1):++k==c&&(k=0);}for(o=new Krb.ueb(a.i);o.a<o.c.a.length;){n=Krb.teb(o);if(n.length==c){d=true;for(h=0;h<c;h++){if(n[h]!=p[h]){d=false;break;}}if(d)return;}}Krb.Ei(a.i,p);m=Jrb.Ar(a,p);Krb.Ei(a.j,m);Jrb.Rr(a,p,m);};Jrb.pr=function pr(a,b,c){var d,e,f,g,h,i,j;i=tX(Drb.NY,Qsb,6,a.f,15,1);g=tX(Drb.NY,Qsb,6,a.f,15,1);j=tX(Drb.c3,itb,6,a.g.f,16,1);d=Jrb.Ik(a.g,0,b);e=Jrb.Ik(a.g,1,b);i[0]=d;i[1]=e;g[1]=-1;j[e]=true;h=1;while(h>=1){++g[h];if(g[h]==Jrb.Gn(a.g,i[h])){j[i[h]]=false;--h;continue;}f=Jrb.Fn(a.g,i[h],g[h]);if(j[f]||c[f])continue;if(f==d&&h>1){Jrb.or(a,i,h+1);if(a.i.a.length>=vtb)return;continue;}if(h+1<a.f){++h;i[h]=f;j[f]=true;g[h]=-1;}}};Jrb.qr=function qr(a,b,c,d,e,f,g,h){var i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D;A=Krb.Ji(a.i,b);for(l=A,m=0,n=l.length;m<n;++m){k=l[m];if(!Jrb.Nr(a,k))return true;}B=Krb.Ji(a.j,b);C=B.length;o=0;j=0;D=false;for(v=0;v<C;v++){o<<=1;j<<=1;if(Jrb.Pr(a,B[v])){o|=1;}else if(h&&Jrb.Mr(a,B[v])){o|=1;j|=1;}else {i=c[b][v];if(i!=-1){if(d[i]){if(e[i]){o|=1;f[i]||(j|=1);}}else {D=true;}}}}u=false;switch(C){case 3:p=BX(nX(Drb.NY,1),Qsb,6,15,[2,1,4]);u=true;for(t=0;t<3;t++){if((o&p[t])==p[t]){if(Jrb.Dk(a.g,A[t])==6&&Jrb.mk(a.g,A[t])==1||Jrb.Dk(a.g,A[t])==5&&Jrb.mk(a.g,A[t])==0){e[b]=true;g[b]=t;(j&p[t])==0&&(u=false);}}}break;case 5:q=BX(nX(Drb.NY,1),Qsb,6,15,[10,5,18,9,20]);u=true;for(w=0;w<5;w++){if((o&q[w])==q[w]){switch(Jrb.Dk(a.g,A[w])){case 6:if(Jrb.mk(a.g,A[w])==-1){e[b]=true;g[b]=w;(j&q[w])==0&&(u=false);}break;case 7:if(Jrb.mk(a.g,A[w])<=0){e[b]=true;g[b]=w;}break;case 8:e[b]=true;g[b]=w;break;case 16:case 34:case 52:if(Jrb.Gn(a.g,A[w])==2){e[b]=true;g[b]=w;}}}}break;case 6:u=true;if((o&21)==21){e[b]=true;(j&21)==0&&(u=false);}if((o&42)==42){e[b]=true;(j&42)==0&&(u=false);}break;case 7:r=BX(nX(Drb.NY,1),Qsb,6,15,[42,21,74,37,82,41,84]);u=true;for(s=0;s<7;s++){if((o&r[s])==r[s]){if(Jrb.Dk(a.g,A[s])==6&&(Jrb.mk(a.g,A[s])==1||h&&Jrb.Dr(a,A[s]))||Jrb.Dk(a.g,A[s])==5&&Jrb.mk(a.g,A[s])==0){e[b]=true;g[b]=s;(j&r[s])==0&&(u=false);}}}}e[b]&&!u&&(f[b]=true);if(e[b])return true;return !D;};Jrb.rr=function rr(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p,q;f=tX(Drb.NY,lub,7,a.i.a.length,0,2);for(j=0;j<a.i.a.length;j++){f[j]=tX(Drb.NY,Qsb,6,Krb.Ji(a.i,j).length,15,1);for(k=0;k<Krb.Ji(a.i,j).length;k++)f[j][k]=-1;}p=tX(Drb.NY,Qsb,6,a.g.g,15,1);for(n=0;n<a.j.a.length;n++){o=Krb.Ji(a.j,n);if(o.length==3||o.length>=5&&o.length<=7){for(i=0;i<o.length;i++){h=o[i];if(Jrb.Gn(a.g,Jrb.Ik(a.g,0,h))==3&&Jrb.Gn(a.g,Jrb.Ik(a.g,1,h))==3){if(p[h]>0){f[p[h]>>>16][p[h]&32767]=n;f[n][i]=p[h]>>>16;}else {p[h]=(n<<16)+Iub+i;}}}}}g=tX(Drb.c3,itb,6,a.i.a.length,16,1);q=0;l=-1;while(q>l){l=q;for(m=0;m<a.i.a.length;m++){if(!g[m]){if(Jrb.qr(a,m,f,g,b,c,d,e)){g[m]=true;++q;}}}}};Jrb.sr=function sr(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o;e=Jrb.Ik(a.g,0,b);f=Jrb.Ik(a.g,1,b);i=tX(Drb.NY,Qsb,6,a.g.f,15,1);j=tX(Drb.NY,Qsb,6,a.g.f,15,1);k=tX(Drb.NY,Qsb,6,a.g.f,15,1);i[0]=e;i[1]=f;j[e]=1;j[f]=2;k[e]=-1;k[f]=e;h=1;l=1;while(h<=l){for(m=0;m<Jrb.Gn(a.g,i[h]);m++){g=Jrb.Fn(a.g,i[h],m);if(h>1&&g==e){o=tX(Drb.NY,Qsb,6,j[i[h]],15,1);d=i[h];for(n=0;n<o.length;n++){o[n]=d;d=k[d];}return o;}if(j[g]==0&&!c[g]){i[++l]=g;j[g]=j[i[h]]+1;k[g]=i[h];}}++h;}return null;};Jrb.tr=function tr(a,b,c){var d,e;e=Krb.Ji(a.i,b);for(d=0;d<e.length;d++)if(c==e[d])return d;return  -1;};Jrb.ur=function ur(a,b){return a.a[b]&Gsb;};Jrb.vr=function vr(a,b,c){var d,e;e=Krb.Ji(a.j,b);for(d=0;d<e.length;d++)if(c==e[d])return d;return  -1;};Jrb.wr=function wr(a,b){return a.b[b]&Gsb;};Jrb.xr=function xr(a,b){return a.c[b];};Jrb.yr=function yr(a,b){return Krb.Ji(a.i,b);};Jrb.zr=function zr(a,b){return Krb.Ji(a.j,b);};Jrb.Ar=function Ar(a,b){var c,d,e,f,g;f=b.length;g=tX(Drb.NY,Qsb,6,f,15,1);for(d=0;d<f;d++){c=d==f-1?b[0]:b[d+1];for(e=0;e<Jrb.Gn(a.g,b[d]);e++){if(Jrb.Fn(a.g,b[d],e)==c){g[d]=Jrb.Hn(a.g,b[d],e);break;}}}return g;};Jrb.Br=function Br(a,b){return Krb.Ji(a.j,b).length;};Jrb.Cr=function Cr(a,b,c){var d;for(d=0;d<a.j.a.length;d++)if(Jrb.Ir(a,d,b)&&Jrb.Ir(a,d,c))return d;return  -1;};Jrb.Dr=function Dr(a,b){var c;for(c=0;c<Jrb.Gn(a.g,b);c++)if(Jrb.In(a.g,b,c)==2&&Jrb.Dk(a.g,Jrb.Fn(a.g,b,c))==8)return true;return false;};Jrb.Er=function Er(a,b){return a.d[b];};Jrb.Fr=function Fr(a,b){return (a.a[b]&Fsb)!=0;};Jrb.Gr=function Gr(a,b){return (a.b[b]&Fsb)!=0;};Jrb.Hr=function Hr(a,b,c){var d,e;e=Krb.Ji(a.i,b);for(d=0;d<e.length;d++)if(c==e[d])return true;return false;};Jrb.Ir=function Ir(a,b,c){var d,e;e=Krb.Ji(a.j,b);for(d=0;d<e.length;d++)if(c==e[d])return true;return false;};Jrb.Jr=function Jr(a,b){return a.e[b];};Jrb.Kr=function Kr(a,b){return (a.b[b]&oub)!=0;};Jrb.Lr=function Lr(a,b){return (a.a[b]&bub)!=0;};Jrb.Mr=function Mr(a,b){var c,d,e,f,g,h;for(g=0;g<2;g++){c=Jrb.Ik(a.g,g,b);if(Jrb.Dk(a.g,c)==7&&Jrb.Gn(a.g,c)==2){d=Jrb.Ik(a.g,1-g,b);if(Jrb.Dk(a.g,d)==6){for(h=0;h<Jrb.Gn(a.g,d);h++){e=Jrb.Fn(a.g,d,h);f=Jrb.Hn(a.g,d,h);if((Jrb.Dk(a.g,e)==8||Jrb.Dk(a.g,e)==16)&&Jrb.Qk(a.g,f)==2&&Jrb.Gn(a.g,e)==1)return true;}}}}return false;};Jrb.Nr=function Nr(a,b){var c,d,e,f,g;if(a.g.K){if(E3(j3(Jrb.yk(a.g,b),1),0)){return true;}else {g=Jrb.tk(a.g,b);if(g!=null){for(d=g,e=0,f=d.length;e<f;++e){c=d[e];if(c==5||c==6||c==7||c==8||c==15||c==16||c==33||c==34)return true;}return false;}}}return Jrb.Vr(Jrb.Dk(a.g,b));};Jrb.Or=function Or(a,b){var c,d,e,f,g;if(a.g.K){if(E3(j3(Jrb.yk(a.g,b),1),0))return false;c=Jrb.tk(a.g,b);if(c!=null){for(e=c,f=0,g=e.length;f<g;++f){d=e[f];if(!Jrb.Ym(d))return false;}return true;}}return Jrb.Ym(Jrb.Dk(a.g,b));};Jrb.Pr=function Pr(a,b){return Jrb.Qk(a.g,b)>1||Jrb.Tk(a.g,b)==8;};Jrb.Qr=function Qr(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q;for(q=0;q<a.d.length;q++){if(a.d[q]){p=false;for(d=Krb.Ji(a.i,q),f=0,h=d.length;f<h;++f){b=d[f];a.a[b]|=Fsb;Jrb.Or(a,b)&&(p=true);}for(k=Krb.Ji(a.j,q),m=0,o=k.length;m<o;++m){i=k[m];a.b[i]|=Fsb;}if(a.e[q]){for(c=Krb.Ji(a.i,q),e=0,g=c.length;e<g;++e){b=c[e];a.a[b]|=oub;}for(j=Krb.Ji(a.j,q),l=0,n=j.length;l<n;++l){i=j[l];a.b[i]|=oub;}}if(p){for(c=Krb.Ji(a.i,q),e=0,g=c.length;e<g;++e){b=c[e];a.a[b]|=bub;}for(j=Krb.Ji(a.j,q),l=0,n=j.length;l<n;++l){i=j[l];a.b[i]|=bub;}}}}};Jrb.Rr=function Rr(a,b,c){var d,e,f,g;g=b.length;for(f=0;f<g;f++){d=a.a[b[f]]&Gsb;if(d==0||d>g){a.a[b[f]]&=vub;a.a[b[f]]|=g;}}for(e=0;e<g;e++){d=a.b[c[e]]&Gsb;if(d==0||d>g){a.b[c[e]]&=vub;a.b[c[e]]|=g;}}};Jrb.Sr=function Sr(a,b,c){var d;d=Krb.Ji(a.j,b).length;while(c>=d)c-=d;while(c<0)c+=d;return c;};Jrb.Tr=function Tr(a,b){Jrb.Ur.call(this,a,b);};Jrb.Ur=function Ur(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;this.g=a;this.f=7;this.i=new Krb.Yi();this.j=new Krb.Yi();this.a=tX(Drb.NY,Qsb,6,this.g.f,15,1);this.b=tX(Drb.NY,Qsb,6,this.g.g,15,1);this.g.zb(1);m=tX(Drb.c3,itb,6,this.g.f,16,1);n=tX(Drb.c3,itb,6,this.g.g,16,1);do{g=false;for(c=0;c<this.g.f;c++){if(!m[c]){q=0;for(l=0;l<Jrb.Gn(this.g,c);l++)m[Jrb.Fn(this.g,c,l)]||++q;if(q<2){m[c]=true;for(k=0;k<Jrb.Gn(this.g,c);k++)n[Jrb.Hn(this.g,c,k)]=true;g=true;}}}}while(g);s=0;while(s<this.g.f&&m[s])++s;if(s==this.g.f)return;i=tX(Drb.NY,Qsb,6,this.g.f,15,1);i[0]=s;p=tX(Drb.NY,Qsb,6,this.g.f,15,1);p[0]=-1;h=tX(Drb.NY,Qsb,6,this.g.f,15,1);h[s]=1;f=0;j=0;o=1;while(f<=j){for(k=0;k<Jrb.Gn(this.g,i[f]);k++){e=Jrb.Fn(this.g,i[f],k);if(e==p[i[f]])continue;if(h[e]!=0){Jrb.pr(this,Jrb.Hn(this.g,i[f],k),m);continue;}if(!m[e]){h[e]=o;p[e]=i[f];i[++j]=e;}}++f;if(f>j){for(c=0;c<this.g.f;c++){if(h[c]==0&&!m[c]){h[c]=++o;i[++j]=c;p[c]=-1;break;}}}}if((b&4)!=0){this.d=tX(Drb.c3,itb,6,this.i.a.length,16,1);this.e=tX(Drb.c3,itb,6,this.i.a.length,16,1);this.c=tX(Drb.NY,Qsb,6,this.i.a.length,15,1);Jrb.rr(this,this.d,this.e,this.c,(b&8)!=0);Jrb.Qr(this);}if((b&2)!=0){for(d=0;d<this.g.g;d++){if(!n[d]&&Jrb.Qk(this.g,d)!=0){r=Jrb.sr(this,d,m);r!=null&&Jrb.Rr(this,r,Jrb.Ar(this,r));}}}};Jrb.Vr=function Vr(a){return a==5||a==6||a==7||a==8||a==15||a==16||a==33||a==34;};Z3(127,1,{},Jrb.Tr);_.f=0;Drb.uZ=B5(127);Jrb.Wr=function Wr(a,b){var c,d,e,f,g,h,i,j,k,l,m;if(b==2||b==5){Krb.Ei(a.D,Jrb.zs(a.F,a.F.length));!!a.b&&Krb.Ei(a.a,Jrb.As(a.B,a.B.length));}else if(b==4){m=Jrb.gs(Jrb.zs(a.F,a.F.length));if(!Krb.kkb(a.N,m)){Krb.ikb(a.N,m);Krb.Ei(a.D,Jrb.zs(a.F,a.F.length));!!a.b&&Krb.Ei(a.a,Jrb.As(a.B,a.B.length));}}else if(b==3){m=Jrb.gs(Jrb.zs(a.F,a.F.length));if(!Krb.kkb(a.N,m)){l=false;for(j=(h=new Krb.Hjb(new Krb.Njb(new Krb.ydb(a.N.a).a).b),new Krb.Hdb(h));Krb.gcb(j.a.a);){i=(g=Krb.Fjb(j.a),g.mc());k=0;for(d=m,e=0,f=d.length;e<f;++e){c=d[e];while(k<i.length&&i[k]<c)++k;if(k<i.length){if(c==i[k]){l=true;break;}}}if(l)break;}if(!l){Krb.ikb(a.N,m);Krb.Ei(a.D,Jrb.zs(a.F,a.F.length));!!a.b&&Krb.Ei(a.a,Jrb.As(a.B,a.B.length));}}}else if(b==6){m=Jrb.hs(a,Jrb.zs(a.F,a.F.length));if(!Krb.kkb(a.N,m)){Krb.ikb(a.N,m);Krb.Ei(a.D,Jrb.zs(a.F,a.F.length));!!a.b&&Krb.Ei(a.a,Jrb.As(a.B,a.B.length));}}};Jrb.Xr=function Xr(a,b,c){var d,e,f,g,h,i,j,k,l,m;h=Jrb.Gn(a.G,b);d=a.n[c];if(d>h)return false;j=Jrb.yk(a.G,b);f=Jrb.yk(a.g,c);e=Jrb.tk(a.g,c);i=Jrb.tk(a.G,b);if(E3(j3(f,1),0)){if(e!=null){if(E3(j3(j,1),0)){if(i==null)return false;if(!Jrb.ms(e,i))return false;}else {if(i!=null){if(Jrb.os(i,e))return false;}else {if(Jrb.ls(Jrb.Dk(a.G,b),e))return false;}}}}else {if(E3(j3(j,1),0))return false;if(e!=null){if(i!=null){if(!Jrb.ms(i,e))return false;}else {if(!Jrb.ls(Jrb.Dk(a.G,b),e))return false;}}else {if(i!=null)return false;if(a.I[b]!=a.j[c])return false;}}if(E3(G3(j,f),0)){if(E3(j3(f,Vtb),0)){if(a.G.K&&t3(j3(j,Vtb),0))return false;else if(d!=h)return false;}if(E3(j3(f,qtb),0)){if(d>=h&&t3(j3(j,qtb),0))return false;}}if(E3(j3(a.H[b],F3(a.i[c])),0))return false;if(E3(j3(a.A[c],F3(a.L[b])),0))return false;g=j3(f,Usb);if(a.G.K){k=j3(f,Usb);if(o3(k,0)!=0&&(o3(g,0)==0||E3(j3(g,F3(k)),0)))return false;}else {if(o3(g,0)!=0&&t3(j3(g,a.L[b]),0))return false;}if(Jrb.mk(a.g,c)!=0&&Jrb.mk(a.g,c)!=Jrb.mk(a.G,b))return false;if(Jrb.wk(a.g,c)!=0&&Jrb.wk(a.g,c)!=Jrb.wk(a.G,b))return false;if(Jrb.zk(a.g,c)!=0&&Jrb.zk(a.g,c)!=Jrb.zk(a.G,b))return false;l=N3(I3(j3(Jrb.yk(a.g,c),Stb),22));if(l!=0){if(a.G.K){m=N3(I3(j3(Jrb.yk(a.G,b),Stb),22));if(l!=m)return false;}else {if(Jrb.An(a.G,b)!=l)return false;}}return true;};Jrb.Yr=function Yr(a,b,c){var d,e,f,g,h,i,j,k,l,m;j=a.J[b];g=a.k[c];if((Jrb.Sk(a.g,c)&uub)!=0){i=Jrb.Uk(a.G,b);e=Jrb.Uk(a.g,c);f=Jrb.Sk(a.g,c)&31;if(i!=e&&!(i==1&&(f&1)!=0)&&!(i==2&&(f&2)!=0)&&!(i==4&&(f&4)!=0)&&!(i==32&&(f&32)!=0)&&!(i==64&&(f&64)!=0)&&!(i==16&&(f&16)!=0)&&!(i==8&&(f&8)!=0))return false;j&=-32;g&=-32;}if((j&~g)!=0)return false;m=(Jrb.Sk(a.g,c)&Gtb)>>17;if(m!=0){if(a.G.K&&m==(Jrb.Sk(a.G,c)&Gtb)>>17)return true;if(m<=2){k=Jrb.En(a.G,b);return m==1?k>=8&&k<=12:k>=12;}d=false;l=Jrb.jo(a.G);for(h=0;h<l.i.a.length;h++){if(Krb.Ji(l.j,h).length==m){if(Jrb.Ir(l,h,b)){d=true;break;}}}if(!d)return false;}return true;};Jrb.Zr=function Zr(a){var b,c;a.b=null;for(b=0;b<a.g.g;b++){if(Jrb.ll(a.g,b)){!a.b&&(a.b=new Krb.Yi());c=new Jrb.Bs();c.a=Jrb.Ik(a.g,0,b);c.b=Jrb.Ik(a.g,1,b);c.d=Jrb.Kk(a.g,b);c.c=Jrb.Jk(a.g,b);Krb.Ei(a.b,c);}}};Jrb.$r=function $r(a){var b,c,d,e,f,g,h,i,j,k,l,m;Jrb.ou(a.g,a.M);i=$wnd.Math.max(a.g.f,a.g.g)+16;a.r=tX(Drb.NY,Qsb,6,i,15,1);a.t=tX(Drb.NY,Qsb,6,i,15,1);a.u=tX(Drb.NY,Qsb,6,i,15,1);a.s=tX(Drb.c3,itb,6,i+1,16,1);g=tX(Drb.c3,itb,6,a.g.f,16,1);h=tX(Drb.c3,itb,6,a.g.g,16,1);d=0;for(c=0;c<a.g.f;c++){if(!a.C[c]&&!g[c]){a.r[d]=c;a.u[d]=-1;a.t[d]=-1;j=d;while(d<=j){for(k=0;k<Jrb.sn(a.g,a.r[d]);k++)j=Jrb.ws(a,d,j,k,g,h,-1);while(a.s[++d]);}}}a.v=d;if(a.o!=0){j=a.v-1;for(f=0;f<a.d;f++){d=0;while(d<=j){for(l=0;l<Jrb.sn(a.g,a.r[d]);l++)j=Jrb.ws(a,d,j,l,g,h,f);while(a.s[++d]);}}for(b=0;b<a.g.f;b++){if(a.C[b]&&!g[b]){a.r[d]=b;a.u[d]=-1;a.t[d]=-1;j=d;while(d<=j){for(l=0;l<Jrb.sn(a.g,a.r[d]);l++)Jrb.Fn(a.g,a.r[d],l)<a.g.f&&(j=Jrb.ws(a,d,j,l,g,h,a.f[b]));while(a.s[++d]);}}}a.e=tX(Drb.NY,Qsb,6,a.d,15,1);for(m=0;m<a.d;m++)a.e[m]=-1;for(k=a.v;k<d;k++){e=a.f[a.r[k]];a.e[e]==-1&&(a.e[e]=k);}}a.w=d;};Jrb._r=function _r(a,b){var c,d,e,f,g,h,i,j;j=Jrb.jo(a.G);for(h=0;h<j.i.a.length;h++){if(Jrb.Ir(j,h,b)){i=tX(Drb.NY,Qsb,6,2,15,1);for(e=0;e<2;e++){i[e]=xsb;d=Jrb.Ik(a.G,e,b);for(f=0;f<Jrb.Gn(a.G,d);f++){c=Jrb.Fn(a.G,d,f);c!=Jrb.Ik(a.G,1-e,b)&&i[e]>c&&(i[e]=c);}}g=0;Jrb.Hr(j,h,i[0])&&++g;Jrb.Hr(j,h,i[1])&&++g;if(g==2)return 2;if(g==1)return 1;return 2;}}return 0;};Jrb.as=function as(a,b,c){var d,e,f,g,h;if(a.b){a.B=tX(Drb.c3,itb,6,b.length,16,1);for(e=new Krb.ueb(a.b);e.a<e.c.a.length;){d=Krb.teb(e);if(a.f==null||c==-1&&a.f[d.a]==-1&&a.f[d.b]==-1||c!=-1&&(a.f[d.a]==c||a.f[d.b]==c)){h=tX(Drb.NY,Qsb,6,d.c+2,15,1);f=Jrb.co(a.G,h,a.F[d.a],a.F[d.b],d.c+1,b,null)-1;if(f<d.d||f>d.c)return false;for(g=1;g<=f;g++)a.B[h[g]]=true;}}}return true;};Jrb.bs=function bs(a,b){var c,d,e,f,g,h,i,j;for(e=0;e<a.g.g;e++){if((Jrb.Sk(a.g,e)&qub)!=0){f=Jrb.Rk(a.g,e);if(f==0)continue;c=Jrb.Ik(a.g,0,e);d=Jrb.Ik(a.g,1,e);if(a.f==null||b==-1&&a.f[c]==-1&&a.f[d]==-1||b!=-1&&(a.f[c]==b||a.f[d]==b)){g=a.F[c];h=a.F[d];i=Jrb.Dn(a.G,g,h);j=Jrb.Rk(a.G,i);if(j==0){Jrb.Po(a.G,i)&&(j=Jrb._r(a,i));if(j==0)continue;}if(f==3)continue;if(j==3)continue;if(Jrb.is(a,e)==(f==j))return false;}}}return true;};Jrb.cs=function cs(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p;g=0;for(i=0;i<a.g.f;i++){if((a.f==null||a.f[i]==b)&&E3(j3(Jrb.yk(a.g,i),Utb),0)){m=a.F[i];l=Jrb.xk(a.g,i);o=Jrb.xk(a.G,m);if(l==0)continue;if(l==3)continue;if(o==0||o==3)return false;if(Jrb.rk(a.g,i)==1){++g;continue;}if(Jrb.rk(a.G,m)==1)return false;if(Jrb.rk(a.g,i)==2){++g;continue;}if(Jrb.rk(a.G,m)==2)return false;if(Jrb.ns(a,i)==(l==o))return false;}}if(g!=0){e=tX(Drb.NY,Qsb,6,g,15,1);f=0;for(j=0;j<a.g.f;j++){if((a.f==null||a.f[j]==b)&&E3(j3(Jrb.yk(a.g,j),Utb),0)){l=Jrb.xk(a.g,j);l!=0&&l!=3&&(e[f++]=Jrb.qk(a.g,j)<<24|Jrb.rk(a.g,j)<<22|j);}}Hrb.Nmb(e,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));f=0;while(f<e.length){k=e[f]&Nub;n=a.F[k];c=e[f]&-4194304;d=Jrb.ns(a,k)^Jrb.xk(a.g,k)==Jrb.xk(a.G,n);for(++f;f<e.length&&(e[f]&-4194304)==c;f++){h=e[f]&Nub;m=a.F[h];if(Jrb.rk(a.G,m)!=Jrb.rk(a.G,n)||Jrb.qk(a.G,m)!=Jrb.qk(a.G,n))return false;p=Jrb.ns(a,h)^Jrb.xk(a.g,h)==Jrb.xk(a.G,m);if(p!=d)return false;}}}return true;};Jrb.ds=function ds(a,b,c){var d,e,f,g,h,i,j;a.O=false;a.D=new Krb.Yi();a.a=new Krb.Yi();Krb.ijb(a.N.a);if(!a.G||!a.g)return 0;if(a.g.f-a.o>a.G.f||a.g.g-a.p>a.G.g)return 0;if(a.g.f==0)return 0;b==6&&(a.M=63);Jrb.ss(a,c);d=tX(Drb.c3,itb,6,a.G.f,16,1);a.F=tX(Drb.NY,Qsb,6,a.g.f,15,1);Krb.Geb(a.F);h=tX(Drb.NY,Qsb,6,a.w,15,1);Krb.Meb(h,h.length,-1);f=0;while(!a.O){if(a.v!=0){j=a.t[f]==-1?a.G.f:Jrb.sn(a.G,a.F[a.t[f]]);++h[f];if(h[f]==j){h[f]=-1;if(f==0)break;--f;a.s[f]||(d[a.F[a.r[f]]]=false);continue;}if(a.t[f]==-1){if(!d[h[f]]){if(Jrb.Xr(a,h[f],a.r[f])){a.F[a.r[f]]=h[f];d[h[f]]=true;++f;}}}else {if(Jrb.Fn(a.G,a.F[a.t[f]],h[f])>=a.G.f)continue;e=Jrb.Fn(a.G,a.F[a.t[f]],h[f]);if(a.s[f]){e==a.F[a.r[f]]&&Jrb.Yr(a,Jrb.Hn(a.G,a.F[a.t[f]],h[f]),a.u[f])&&++f;}else {if(!d[e]){if(Jrb.Xr(a,e,a.r[f])&&Jrb.Yr(a,Jrb.Hn(a.G,a.F[a.t[f]],h[f]),a.u[f])){d[e]=true;a.F[a.r[f]]=e;++f;}}}}}if(f==a.v){if(Jrb.cs(a,-1)&&Jrb.bs(a,-1)&&Jrb.as(a,d,-1)){i=false;for(g=0;g<a.d;g++){if(Jrb.js(a,d,h,g)){i=true;break;}}if(b==1&&!i)return 1;if(!i){Jrb.Wr(a,b);if(b==2)return 1;}}if(f==0)break;--f;a.s[f]||(d[a.F[a.r[f]]]=false);}}return a.D.a.length;};Jrb.es=function es(a,b){var c,d,e,f,g,h,i,j;i=0;if(a.K){b<a.f&&Jrb.Lr(a.p,b)?i=G3(i,ivb):b<a.f&&Jrb.Fr(a.p,b)&&(i=G3(i,2));j=(e=a.u[b]&Oub,e==0?0:e==vtb?2:e==Vtb?3:4);if(j!=0){i=G3(i,8);j>2&&(i=G3(i,16));j>3&&(i=G3(i,32));}c=a.s[b];c<0?i=G3(i,ztb):c>0&&(i=G3(i,ytb));g=a.j[b];switch(g){case 0:break;case 1:i=G3(i,oub);break;case 2:i=G3(i,Ftb);break;case 3:i=G3(i,Gtb);break;default:i=G3(i,Htb);}d=Jrb.vn(a,b);switch(d){case 0:break;case 1:i=G3(i,Otb);break;case 2:i=G3(i,Ptb);break;case 3:i=G3(i,Qtb);break;default:i=G3(i,Rtb);}h=a.o[b];h>0&&(i=G3(i,ptb));h>1&&(i=G3(i,Iub));}else {b<a.f&&Jrb.Lr(a.p,b)?i=G3(i,ivb):b<a.f&&Jrb.Fr(a.p,b)?i=G3(i,2):i=G3(i,4);(a.u[b]&Hub)!=0?i=G3(i,rtb):i=G3(i,stb);j=(e=a.u[b]&Oub,e==0?0:e==vtb?2:e==Vtb?3:4);j==0?i=G3(i,112):j==2?i=G3(i,104):j==3?i=G3(i,88):i=G3(i,56);c=a.s[b];c==0?i=G3(i,xtb):c<0?i=G3(i,ztb):i=G3(i,ytb);f=a.e[b]-Jrb._n(a,b)+Jrb.Wn(a,b);switch(f){case 0:i=G3(i,1792);break;case 1:i=G3(i,1664);break;case 2:i=G3(i,1408);break;default:i=G3(i,896);}g=a.j[b];switch(g){case 0:i=G3(i,3932160);break;case 1:i=G3(i,3801088);break;case 2:i=G3(i,Ctb);break;case 3:i=G3(i,Dtb);break;default:i=G3(i,Htb);}d=Jrb.vn(a,b);switch(d){case 0:i=G3(i,Jtb);break;case 1:i=G3(i,Ktb);break;case 2:i=G3(i,Ltb);break;case 3:i=G3(i,Mtb);break;default:i=G3(i,Rtb);}h=a.o[b];switch(h){case 0:i=G3(i,98304);break;case 1:i=G3(i,81920);break;default:i=G3(i,49152);}}return i;};Jrb.fs=function fs(a,b){var c;c=0;if(b<a.g&&(Jrb.Kr(a.p,b)||a.H[b]==8)||a.H[b]==8)c|=8;else switch(Jrb.Qk(a,b)){case 0:c|=16;break;case 1:c|=1;break;case 2:c|=2;break;case 3:c|=4;break;case 4:c|=32;break;case 5:c|=64;}(a.F[b]&64)!=0?c|=256:a.K||(c|=128);b<a.g&&Jrb.Gr(a.p,b)?c|=Etb:a.K||(c|=jvb);return c;};Jrb.gs=function gs(a){var b,c,d,e,f,g,h,i,j,k;i=0;for(d=a,f=0,h=d.length;f<h;++f){b=d[f];b==-1&&++i;}if(i!=0){k=a;a=tX(Drb.NY,Qsb,6,k.length-i,15,1);j=0;for(c=k,e=0,g=c.length;e<g;++e){b=c[e];b!=-1&&(a[j++]=b);}}Hrb.Nmb(a,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));return a;};Jrb.hs=function hs(a,b){var c,d,e,f,g,h,i,j;g=0;for(d=b,e=0,f=d.length;e<f;++e){c=d[e];c==-1&&++g;}j=tX(Drb.NY,Qsb,6,b.length-g,15,1);i=0;for(h=0;h<b.length;h++){if(b[h]!=-1){j[i]=Jrb.zu(a.g,h)<<16|Jrb.zu(a.G,b[h]);++i;}}Hrb.Nmb(j,Hrb.rnb(a4(Krb.ofb.prototype.pc,Krb.ofb,[])));return j;};Jrb.is=function is(a,b){var c,d,e,f,g,h,i,j,k,l,m,n;f=false;for(e=0;e<2;e++){c=Jrb.Ik(a.g,e,b);k=a.F[c];if(Jrb.Gn(a.G,k)>2){m=Jrb.Ik(a.g,1-e,b);i=xsb;for(h=0;h<Jrb.Gn(a.g,c);h++){d=Jrb.Fn(a.g,c,h);d!=m&&i>d&&(i=d);}n=a.F[m];j=xsb;for(g=0;g<Jrb.Gn(a.G,k);g++){l=Jrb.Fn(a.G,k,g);l!=n&&j>l&&(j=l);}a.F[i]!=j&&(f=!f);}}return f;};Jrb.js=function js(a,b,c,d){var e,f,g,h,i,j,k,l;h=a.e[d];i=h+1;while(i<a.w&&a.f[a.r[i]]==d)++i;for(k=h;k<i;k++)c[k]=-1;g=h;while(true){l=a.t[g]==-1?a.G.f:Jrb.sn(a.G,a.F[a.t[g]]);++c[g];if(c[g]==l){c[g]=-1;if(g==h)break;--g;if(!a.s[g]){b[a.F[a.r[g]]]=false;a.F[a.r[g]]=-1;}continue;}if(a.t[g]==-1){if(!b[c[g]]){if(Jrb.Xr(a,c[g],a.r[g])){a.F[a.r[g]]=c[g];b[c[g]]=true;++g;}}}else {if(Jrb.Fn(a.G,a.F[a.t[g]],c[g])>=a.G.f){++c[g];continue;}f=Jrb.Fn(a.G,a.F[a.t[g]],c[g]);if(a.s[g]){f==a.F[a.r[g]]&&Jrb.Yr(a,Jrb.Hn(a.G,a.F[a.t[g]],c[g]),a.u[g])&&++g;}else {if(!b[f]){if(Jrb.Xr(a,f,a.r[g])&&Jrb.Yr(a,Jrb.Hn(a.G,a.F[a.t[g]],c[g]),a.u[g])){b[f]=true;a.F[a.r[g]]=f;++g;}}}}if(g==i){if(Jrb.cs(a,d)&&Jrb.bs(a,d)&&Jrb.as(a,b,d)){for(j=h;j<i;j++){if(!a.s[j]){e=a.r[j];b[a.F[e]]=false;a.F[e]=-1;}}return true;}--g;if(!a.s[g]){b[a.F[a.r[g]]]=false;a.F[a.r[g]]=-1;}}}return false;};Jrb.ks=function ks(a){return Jrb.ds(a,1,a.c)>0;};Jrb.ls=function ls(a,b){var c;for(c=0;c<b.length;c++)if(b[c]==a)return true;return false;};Jrb.ms=function ms(a,b){var c,d,e;e=0;for(d=0;d<a.length;d++){c=a[d];while(b[e]<c){++e;if(e==b.length)return false;}if(b[e]>c)return false;}return true;};Jrb.ns=function ns(a,b){var c,d,e,f,g,h,i,j;g=false;if(Jrb.wn(a.g,b)==0){for(f=1;f<Jrb.Gn(a.g,b);f++){for(h=0;h<f;h++){d=Jrb.Fn(a.g,b,f);e=Jrb.Fn(a.g,b,h);a.F[d]>a.F[e]^d>e&&(g=!g);}}}else {for(f=0;f<Jrb.Gn(a.g,b);f++){c=Jrb.Fn(a.g,b,f);j=0;i=tX(Drb.NY,Qsb,6,3,15,1);for(h=0;h<Jrb.Gn(a.g,c);h++){i[j]=Jrb.Fn(a.g,c,h);i[j]!=b&&++j;}j==2&&a.F[i[0]]>a.F[i[1]]^i[0]>i[1]&&(g=!g);}}return g;};Jrb.os=function os(a,b){var c,d,e,f;e=0;f=0;while(e<a.length&&f<b.length){c=a[e];d=b[f];if(c==d)return true;c<d?++e:++f;}return false;};Jrb.ps=function ps(a,b){var c,d,e,f,g;if(!b||b.q==0||!b.K){a.g=null;return;}a.g=b;a.q=false;Jrb.ou(a.g,1);a.M=7;for(d=0;d<a.g.f;d++)E3(j3(Jrb.yk(a.g,d),{l:Utb,m:0,h:3}),0)&&(a.M=15);for(g=0;g<a.g.g;g++)(Jrb.Sk(a.g,g)&qub)!=0&&(a.M=15);a.K&&a.M!=7&&Jrb.ou(a.G,a.M);a.o=0;a.p=0;a.C=tX(Drb.c3,itb,6,a.g.f,16,1);for(e=0;e<a.g.f;e++){a.C[e]=E3(j3(Jrb.yk(a.g,e),jtb),0);a.C[e]&&++a.o;}a.d=0;a.f=null;if(a.o!=0){for(f=0;f<a.g.g;f++)(a.C[Jrb.Ik(a.g,0,f)]||a.C[Jrb.Ik(a.g,1,f)])&&++a.p;for(c=0;c<a.g.q;c++)Jrb._l(a.g,c,a.C[c]);a.f=tX(Drb.NY,Qsb,6,a.g.q,15,1);a.d=Jrb.Pn(a.g,a.f,true,false);}};Jrb.qs=function qs(a,b,c){Jrb.rs(a,c);Jrb.ps(a,b);};Jrb.rs=function rs(a,b){if(!b||b.q==0){a.G=null;return;}a.G=b;a.K=false;Jrb.ou(a.G,1);};Jrb.ss=function ss(a,b){if(!a.K){Jrb.vs(a,b);a.K=true;}if(!a.q){Jrb.ts(a,b);Jrb.$r(a);Jrb.Zr(a);a.q=true;}};Jrb.ts=function ts(a,b){var c,d,e,f,g,h,i,j,k,l;f=null;i=null;g=null;Jrb.ou(a.g,a.M);a.n=tX(Drb.NY,Qsb,6,a.g.f,15,1);for(d=0;d<a.g.f;d++)a.n[d]=Jrb.Gn(a.g,d);if(a.o!=0){j=new Jrb.Hu(a.g.q,a.g.r);l=tX(Drb.c3,itb,6,a.g.q,16,1);for(e=0;e<a.g.q;e++)l[e]=!a.C[e];Jrb.hn(a.g,j,l,true,null);Jrb.ou(j,a.M);Jrb.us(a,j,b);f=a.i;i=a.k;g=a.j;k=0;for(c=0;c<a.g.f;c++)a.C[c]||(a.n[c]=Jrb.Gn(j,k++));}Jrb.us(a,a.g,b);if(a.o!=0){k=0;for(c=0;c<a.g.q;c++){if(!a.C[c]){a.i[c]=f[k];a.j[c]=g[k++];}}k=0;for(h=0;h<a.g.r;h++){!a.C[Jrb.Ik(a.g,0,h)]&&!a.C[Jrb.Ik(a.g,1,h)]&&(a.k[h]=i[k++]);}}};Jrb.us=function us(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;p=b.f;a.i=tX(Drb.OY,iub,6,b.f,14,1);a.j=tX(Drb.NY,Qsb,6,b.f,15,1);for(h=0;h<p;h++){a.i[h]=P3(j3(G3(Jrb.es(b,h),b.B[h]),kvb),lvb);a.j[h]=b.C[h];(c&1)!=0&&(a.j[h]+=b.s[h]+16<<8);(c&2)!=0&&(a.j[h]+=b.A[h]<<16);}a.A=tX(Drb.OY,iub,6,b.f,14,1);r=(Jrb.ou(b,7),b.p);for(o=0;o<r.i.a.length;o++){n=false;for(j=Krb.Ji(r.j,o),k=0,l=j.length;k<l;++k){m=j[k];if((b.G[m]&Xtb)!=0){n=true;break;}}if(!n){s=Krb.Ji(r.j,o).length;for(e=Krb.Ji(r.i,o),f=0,g=e.length;f<g;++f){d=e[f];s==3?a.A[d]=G3(a.A[d],Wsb):s==4?a.A[d]=G3(a.A[d],Xsb):s==5?a.A[d]=G3(a.A[d],Ysb):s==6?a.A[d]=G3(a.A[d],Zsb):s==7&&(a.A[d]=G3(a.A[d],$sb));}}}q=b.g;a.k=tX(Drb.NY,Qsb,6,b.g,15,1);for(i=0;i<q;i++){a.k[i]=(Jrb.fs(b,i)|b.G[i])&6291967^6291840;(c&4)!=0?(a.k[i]&2)!=0&&(a.k[i]|=8):(c&8)!=0&&(a.k[i]&2)!=0&&i<b.g&&Jrb.Gr(b.p,i)&&(a.k[i]|=8);}};Jrb.vs=function vs(a,b){var c,d,e,f,g,h,i,j,k,l,m,n;Jrb.ou(a.G,a.M);k=a.G.f;a.I=tX(Drb.NY,Qsb,6,k,15,1);a.H=tX(Drb.OY,iub,6,k,14,1);for(g=0;g<k;g++){a.H[g]=P3(j3(G3(Jrb.es(a.G,g),Jrb.yk(a.G,g)),kvb),lvb);a.I[g]=Jrb.Dk(a.G,g);(b&1)!=0&&(a.I[g]+=Jrb.mk(a.G,g)+16<<8);(b&2)!=0&&(a.I[g]+=Jrb.wk(a.G,g)<<16);}a.L=tX(Drb.OY,iub,6,k,14,1);m=Jrb.jo(a.G);for(j=0;j<m.i.a.length;j++){n=Krb.Ji(m.j,j).length;for(d=Krb.Ji(m.i,j),e=0,f=d.length;e<f;++e){h=d[e];n==3?a.L[h]=G3(a.L[h],Wsb):n==4?a.L[h]=G3(a.L[h],Xsb):n==5?a.L[h]=G3(a.L[h],Ysb):n==6?a.L[h]=G3(a.L[h],Zsb):n==7&&(a.L[h]=G3(a.L[h],$sb));}}for(c=0;c<k;c++){n=Jrb.An(a.G,c);n==0?a.L[c]=G3(a.L[c],Vsb):n>7&&(a.L[c]=G3(a.L[c],_sb));}l=a.G.g;a.J=tX(Drb.NY,Qsb,6,l,15,1);for(i=0;i<l;i++)a.J[i]=(Jrb.fs(a.G,i)|Jrb.Sk(a.G,i))&6422527^6291840;};Jrb.ws=function ws(a,b,c,d,e,f,g){var h,i;h=Jrb.Fn(a.g,a.r[b],d);if((!a.C[h]||a.f[h]==g)&&h!=a.t[b]){i=Jrb.Hn(a.g,a.r[b],d);if(!f[i]&&!Jrb.ll(a.g,i)){a.r[++c]=h;a.t[c]=a.r[b];a.u[c]=i;f[i]=true;e[h]?a.s[c]=true:e[h]=true;}}return c;};Jrb.xs=function xs(){this.c=8;this.N=new Krb.okb(new Zrb.hV());};Jrb.ys=function ys(){this.c=1;this.N=new Krb.okb(new Zrb.hV());};Jrb.zs=function zs(a,b){var c;c=tX(Drb.NY,Qsb,6,b,15,1);Erb.Y7(a,0,c,0,$wnd.Math.min(a.length,b));return c;};Jrb.As=function As(a,b){var c;c=tX(Drb.c3,itb,6,b,16,1);Erb.Y7(a,0,c,0,$wnd.Math.min(a.length,b));return c;};Z3(66,1,{},Jrb.xs,Jrb.ys);_.c=0;_.d=0;_.o=0;_.p=0;_.q=false;_.v=0;_.w=0;_.K=false;_.M=0;_.O=false;Drb.xZ=B5(66);Jrb.Bs=function Bs(){};Z3(184,1,{184:1},Jrb.Bs);_.a=0;_.b=0;_.c=0;_.d=0;Drb.vZ=B5(184);Jrb.Es=function Es(){Jrb.Es=_3;Jrb.Cs=BX(nX(Drb.B0,1),Esb,2,6,['QM@HzAmdqjF@','RF@Q``','qC`@ISTAlQE`','`J@H','QM@HzAmdqbF@','qC`@ISTAlQEhqPp@','sJP@DiZhAmQEb','RF@QPvR@','QM@HzA@','qC`@ISTAlQEhpPp@','qC`@Qz`MbHl','sJP@DiZhAmQEcFZF@','RFPDXH','qC`@IVtAlQE`','QM@HvAmdqfF@','sGP@DiVj`FsDVM@','`L@H','sJP@DizhAmQEcFBF@','sJP@DjvhAmQEb','sFp@DiTt@@AlqEcP','sGP@LdbMU@MfHlZ','QMHAIhD','QM@HzAy@','sJP@DkVhAmQEb','sNp@DiUjj@[\\QXu`','sJP@DiZhAmQEcFBF@','sGP@DjVj`FsDVM@','RFPDTH','RG@DXOH@','sGP@Divj`FsDVMcAC@','sGP@Dj}j`FsDVM@','qC`@Qz`MbHmFRF@','sNp@LdbJjj@[\\QXu`','QMHAIhGe@','QM@HzAyd`','QM`AIhD','qC`@ISTA@','sGP@DkUj`FsDVM@','qC`@IVtAlQEhqPp@','sNp@DiUjj@[\\QXuqea`@','KAx@@IRjuUPAlHPfES\\','QM`BN`P','sJP@DjZhAmQEcFJF@','Hid@@DjU^nBBH@FtaBXUMp`','sNp@Diujj@[\\QXuq`a`@','sJP@DjvhAmQEcFZF@','sJP@DjZhAmQEcFFF@','sOp@DjWkB@@FwDVM\\YhX@','sNp@Dj}Zj@[\\QXu`','sNp@DiWjj@[\\QXuq`a`@','sOp@DjWkB@@D','KAx@@ITouUPAlHPfES\\','KAx@@YIDTjjh@vDHSBin@','sNp@DkUZj@[\\QXu`','RFPDXOH@','QM`BN`^L`','qC`@ISTAy@','sGP@LdbMU@MfHl[FVF@','qCb@AIZ`H','KAx@@IRjuUPAlHPfES]FFa`@','KAx@@ITnuUPAlHPfES\\','HiD@@DiUVjj`AmHPfES\\H','sNp@DjUjj@[\\QXu`','sJP@DkVhAmQEcFJF@','sGP@DjVj`FsDVMcCC@','qC`@Qz`MbHmFBF@','sJP@DkfhAmQEb','qC`@IVtAlQEhsPp@','sGP@Djuj`FsDVM@','sGP@Dj}j`FsDVMcMC@','sJP@DiZhA@','KAx@@ISjuUPAlHPfES]F@a`@','sJP@DjZhAmQEcFRF@','KAx@@IRnuUPAlHPfES]F@a`@','HiD@@DjWvjj`AmHPfES\\H','QMHAIhGd@','sNp@DiUjj@[\\QXuq`a`@','KAx@@IVjmUPAlHPfES\\','sGP@DjVj`FsDVMcMC@','QM`AIhGe@','HiD@@LdbJRjjh@[RDIaTwB','qCp@AIZ`H','sGP@LdbMU@MfHl[FFF@','QMDARVA@','sNp@LdbJjj@[\\QXuqba`@','sNp@LdbJjj@[\\QXuqca`@','sGP@Dkej`FsDVM@','qCb@AIZ`OI@','HaD@@DjUZxHH@AlHPfES]FLa`@','sGP@DkYj`FsDVM@','qCb@AIV`H','sNp@LdbJjj@[\\QXuqea`@','sGP@DkUj`FsDVMcEC@','sFp@DiTt@@Axa@','Hmt@@DjU_ZxHHj@AmhPfES\\Lj','QM`BN`^P','qCb@AIZ`OH`','sFp@DiTt@@AxaP','sGP@Djuj`FsDVMcEC@','sGP@Djuj`FsDVMcIC@','sGP@DkUj`FsDVMcKC@','sJP@DkfhAmQEcFRF@','sGP@DjVj`FsDVMcIC@','HaD@@DjUZxHH@AlHPfES]FFa`@','qC`@IRtDVqDV@','sNp@Dj}Zj@[\\QXuqfa`@','KAx@@ITnuUPAlHPfES]FFa`@','HiD@@DkUUjj`AmHPfES\\H','sJQ@@dkU@H','qC`@Qz`H','KAx@@IUkmUPAlHPfES\\','KAx@@ITouUPAlHPfES]FJa`@','sJP@H~j@[TQX`','sGP@DjZj`FsDVM@','sJP@DkVhAmQEcFFF@','sJX@@eKU@H','sJP@DizhAy@','QMHAIhGbP','KAx@@ITouUPAlHPfES]FNa`@','HaD@@DjUZxHD@AlHPfES\\','HaD@@DjUZxHH@A@','sNp@LdbJjj@[\\QXuqaa`@','Hed@@LdbRQUUUP@vTHSBinFP','KAx@@ITouUPAlHPfES]FLa`@','sNp@DkUZj@[\\QXuqba`@','KAx@@ITjuUPAlHPfES]FNa`@','KAx@@YIDTjjh@vDHSBincGPp@','HaD@@DjYvxH`@AlHPfES]FLa`@','RF@QP`','qCb@AIj`H','sNp@DjUjj@[\\QXuqaa`@','sNp@DkVZj@[\\QXu`','KAx@@YIDUJjh@vDHSBin@','sGP@DkYj`FsDVMcIC@','sGP@DjVj`FsDVMcAC@','sGP@DiVj`D','sJP@DkVhAmQEcFZF@','sNp@LdbLjj@[\\QXu`','QM@HvAmdqbF@','HaD@@DjWjXHB@AlHPfES\\','sNp@DjwZj@[\\QXuqba`@','sNp@LdbJjj@[\\QXuqda`@','sFp@DiTt@@Axa`','HiD@@Djuujj`AmHPfES\\H','sNp@DkUZj@[\\QXuqca`@','sJP@DiZhAy@','KAx@@YIDTjjh@vDHSBincCPp@','KAx@@IWNmUPAlHPfES\\','KAx@@IVkMUPAlHPfES\\','sJQ@@dju@H','qCb@AIZ`OH@','qC`@ISTAxa@','sNp@DjyZj@[\\QXu`','Hid@@DjUfaBB`@FtaBXUMp`','HiD@@DiUVjj`AmHPfES\\LXBF@','KAx@@IUjmUPAlHPfES\\','HiD@@DjWvjj`AmHPfES\\LXjF@','sJP@DjVhAmQEb','qCb@AIV`OH`','HiD@@LdbJRjjh@[RDIaTwCFDa`@','KAx@@YIDTjjh@vDHSBinc@Pp@','sNp@DjUjj@[\\QXuqda`@','qC`@Qz`OED','sJP@DkfhAmQEcFZF@','KAx@@YIDbjjh@vDHSBincDPp@','sGP@Djyj`FsDVMcMC@','KAx@@IVrmUPAlHPfES\\','qCp@AIZ`OI@','sJX@@dkU@H','sJQ@@dkU@OH`','sNp@Di]ZjBBvxbqk@','Hkl@@DjU_Uk``bj`@[VDIaTwCJzX','sGP@DjZj`FsDVMcEC@','Hid@@DjU^nBBH@FtaBXUMpqcHX@','sNp@DkeZj@[\\QXu`','sNp@DjYjj@[\\QXuqca`@','sGQ@@djuT@`','HiD@@LdbJTjjh@[RDIaTwB','sOp@DjWkB@@Gd`','HeT@@LdbbRKBDQD@CYPaLJfxY@','qCr@XIKTA@','HiD@@DjW^jj`AmHPfES\\LXJF@','HeT@@DjU]k``b`@[JDIaTwCH','sGP@Djuj`FsDVMcCC@','`IH`B','sOp@DjWkB@@GdX','sJQ@@eKU@H','KAx@@YIDUJjh@vDHSBincBPp@','sJX@@eKU@OH@','KAx@@YIDTjjh@vDHSBincAPp@','sOq@@drm\\@@@`','KAx@@IUkMUPAlHPfES\\','qCp@AIj`H','Hed@@DjUUjjj@FraBXUMpr','sGX@@eJuT@`','sGP@DkUj`FsDVMcCC@','HiD@@Dj}Ujj`AmHPfES\\LXrF@','KAx@@ITouUPAlHPfES]FHa`@','Hed@@DjWujjj@FraBXUMpsFIa`@','sGP@DiUj``mfHlZ','sFp@DiTvjhAlqEcP','Hid@@DjU^nBBH@FtaBXUMpq`XX@','sJP@DkVdAmQEb','qCp@AIZ`OH`','QMhDRVA@','qC`@ISJAlQE`','qCp@BOTAyhl','sJX@@eOU@ODB','sFp@DiTt@@AyaB','sGP@DkUj`FsDVMcMC@','Hid@@DjYUaBH`@FtaBXUMpqcHX@','qC`@Qz`OH@','HiD@@DjUVjj`AmHPfES\\LXZF@','sJP@H~j@[TQXqda`@','sJX@@eKU@OI@','sNp@Djejj@[\\QXu`','sJQ@@dsU@H','sJQ@@dkU@OI`','KAx@@YIMDVjh@vDHSBin@','Hid@@DjU^nBBD@FtaBXUMp`','sNp@DkgZj@[\\QXuqca`@','qC`@IRtDVqDVcEC@','Hed@@LdbRQeUUP@vTHSBinFP','sNp@DiUjj@P','qC`@IRtDT','sNp@DkYZj@[\\QXuqca`@','KAx@@IUkmUPAlHPfES]FDa`@','KAx@@IVjmUPAlHPfES]FNa`@','sOx@@drm\\@@@`','KAx@@ITjuUPAlHPfES]FBa`@','QMDARVAyH','sJP`@dfvhA@','HeT@@DjU_k``b`@[JDIaTwCLXfF@','KAx@@IToUUPAlHPfES]FJa`@','sGP@DkYj`FsDVMcEC@','qCb@AIZ`ODH','`I@`B','KAx@@IUzmUPAlHPfES]FFa`@','sNp@DkfZj@[\\QXu`','KAx@@ITnuUPAlHPfES]F@a`@','HiD@@LddURjjh@[RDIaTwB','sNp@Dj~Zj@[\\QXuqfa`@','Hed@@Dj{uZjj@FraBXUMpr','KAx@@ITsUUPAlHPfES\\','Hid@@LdbRQk``b@AmHPfES\\LXrF@','sOp@DjWkB@@GdH','sJQ@@dkU@OH@','Hid@@DjU^nBBH@FtaBXUMpqahX@','sGP@DiYj``mfHlZ','KAx@@IToUUPAlHPfES]FLa`@','qCp@AJZ`ODH','Hmt@@DjU]ZxHHj@AmhPfES\\Lj','sGP@DkUjPFsDVM@','qC`@IVtA@','Hed@@LdbJReUUP@vTHSBinFP','sNp@DjuZj@[\\QXuqea`@','KAx@@IUkmUPAlHPfES]FNa`@','HiD@@DkVUjj`AmHPfES\\H','Hed@@DkUeZjj@FraBXUMpr','sNp@DkVZj@[\\QXuqea`@','sJP@DiVhHKZbKFLLL@','HiD@@Djuyjj`AmHPfES\\H','sNp@DjUjj@[\\QXuq`a`@','HeT@@DjYUXPbH`@[JDIaTwCH','HiD@@DjwUjj`AmHPfES\\LXRF@','sNq@@djmUPB','KAx@@YIEEZjh@vDHSBincCPp@','sGP@Di^V`dmfHlZ','Hid@@DjYUaBHP@FtaBXUMp`','sNp@DjYjj@[\\QXuqba`@','sGP@Dkej`FsDVMcKC@','HeT@@DjU^k``b`@[JDIaTwCH','qC`@Qv`MbHmFBF@','sGQ@@djmT@`','qCr@XIKTAyH','qC`@IVtAlQEhpPp@','Hid@@LdbbQxXF@@AmHPfES\\LXjF@','sGP@DkYj`FsDVMcCC@','KAx@@IVsMUPAlHPfES\\','qCp@AIj`ODl','HiD@@DkeUjj`AmHPfES\\H','deT@@DjU[k``b`@vTHSBinFP','sJP@DkVdAmQEcFRF@','HiD@@LdbJTjjh@[RDIaTwCFDa`@','HiD@@DkYyjj`AmHPfES\\H','sJP@DjZhAyH','KAx@@IVkMUPAlHPfES]FDa`@','sJX@@dkU@OI@','Hed@@LdbRQUUUP@vTHSBinFXpLL@','Hed@@DjuUZjj@FraBXUMpr','sGP@Djfj`FsDVMcKC@','sNp@DkVZj@[\\QXuqba`@','sNp@DjyZj@[\\QXuqfa`@','qCb@AIj`OH@','sNp@DjUZj@[\\QXu`','KAx@@IWOMUPAlHPfES\\','Hid@@DjU^nBBH@D','Hed@@DjuvZjj@FraBXUMpr','sJP@DiVhHKZbKFLtL@','dmt@@DjU_ZxHHj@C[PaLJfxYVLKC@','sNp@DjuZj@[\\QXuqca`@','sJP@DkfhAmQEcFJF@','sNp@LdbJZj@[\\QXu`','HeT@@DjU_k``b`@[JDIaTwCLXFF@','KAx@@IVlmUPAlHPfES]FNa`@','HeT@@LdbbRKBDQD@CYPaLJfxYcEPp@','Hid@@DjUZnBBH@FtaBXUMpqcHX@','qCa@CIKTA@','HiD@@Dj~]jj`AmHPfES\\LXFF@','sKP@Di\\Zj@[TQX`','sGP@Djfj`FsDVMcEC@','HiD@@DkgYjj`AmHPfES\\H','sNp@DjuZj@[\\QXuqaa`@','KAx@@YIMDVjh@vDHSBincDPp@','sJP@DjVhHKZbKFLTL@','Hid@@LdbRQk``b@AmHPfES\\LXZF@','HiD@@Dj}Ujj`AmHPfES\\LXzF@','HeT@@DjU_k``bP@[JDIaTwCH','sNp@DkUZi@[\\QXu`','HiD@@DjYfjj`AmHPfES\\H','sGP@DjZj`FsDVMcAC@','Hmt@@DjU_jxHHj@AmhPfES\\Lj','Hid@@LdbRQk``R@AmHPfES\\H','KAx@@YIDUJjh@vDHSBincDPp@','qCr@XIKTAyD','sOq@@drm\\@@@|`@','Hed@@DjW^jjj@FraBXUMpsFBa`@','HeT@@DjY]zXFB@@[JDIaTwCH','Hkl@@DjU_Vk``bj`@[VDIaTwCJzX','Hid@@DjY}nBHH@FtaBXUMpqcHX@','sGX@@eKuT@|d@','sGP@Dj^Y`FsDVM@','HcL@@DjU_ZnBBJh@FqaBXUMprn`','sJP@DkVdAmQEcFJF@','sOq@@drm\\@@@|b@','sNp@DjyZj@[\\QXuqaa`@','HaD@@DjUZxHH@AyD@','qC`@Qv`H','dmt@@DjU_ZxHHj@C[PaLJfxYVLYC@','sGP@Dkej`FsDVMcMC@','Hed@@DjUUjjj@FraBXUMpsFHa`@','HeT@@LdbbRkBDQD@CYPaLJfxY@','KAx@@IU{MUPAlHPfES]FLa`@','RG@DTH','sJY@DDeVhA@','KAx@@YIDUJjh@vDHSBinc@Pp@','sJX@@dkU@OI`','sJQ@@dju@OI`','HeT@@LdbbRKBDQD@CYPaLJfxYcFPp@','sFp@DiTvjhAlqEcXpPp@','HaD@@DjUZxHH@AyG@','sNx@@eJ}UPB','sNp@LddUjj@[\\QXuqca`@','HaDH@@RVU[j@@@D','sNp@DkgZi@[\\QXu`','sGY@LDeVj`D','sNp@LdbJfZBZvxbqk@','sJP`@dfvhAyL','sGX@AddQjhAxe`','Hmt@@DjU_ZxHHj@AmhPfES\\LkFIa`@','qCh@CIKTA@','sNp@LdbLjj@[\\QXuq`a`@','sOq@@drm\\@@@|a@','KAx@@IUzmUPAlHPfES]FJa`@','sNx@AddQUUPB','sGP@Di]jP`mfHlZ','sJP`@TeZhA@','KAx@@IRjmUPHKXPaLJfx','HeT@@LdbRTM\\DDT@CYPaLJfxY@','HaF@@@Rfu[j@@@D','Hid@@DjYUaBH`@FtaBXUMpqchX@','KAx@@IUjmTpAlHPfES\\','Hid@@DjU^nBBD@FtaBXUMpqcHX@','sGP@DiUj``mfHl[FFF@','KAx@@IUvmUPAlHPfES]FLa`@','Hed@@LdbQTUUUP@vTHSBinFXqDL@','sJP@DkVhA@','sOx@@drm\\@@@|b@','KAx@@IUkMUPAlHPfES]FDa`@','HeT@@LdbRQU\\DDT@CYPaLJfxY@','HiD@@Dj}Yjj`AmHPfES\\LXrF@','HiD@@Dj{ujj`AmHPfES\\LXFF@','KAx@@IWNmUPAlHPfES]FFa`@','KAx@@IRkMUPHKXPaLJfx','sJP@DjYdAmQEcFZF@','sJY@LDeZhAyL','HaDH@@RVU[f@@@D','sJP`@deVhAyB','HaD@@DjWjZjj`AlHPfES\\','sGP@DkYj`FsDVMcMC@','sNp@DkgZj@[\\QXuqea`@','sJQ@@dlu@H','HeT@@DjU]k``b`@[JDIaTwCLXrF@','sJX@@dkU@OH`','RFDDQFCr`','sJP@DiYXIKZbKFLLL@','KAx@@YIHjjjh@vDHSBincGPp@','Hk\\@@DjU^ukmLHH@@@AmXPfES\\Lki`','sGQ@@djmT@|b@','Hid@@DjUfaBB`@FtaBXUMpqahX@','sNx@@eRmUPB','Hmt@@LdbRVak``ah@FvaBXUMprh','qCr@XIJtA@','KAx@@IWMmUPAlHPfES]FNa`@','HeT@@DjYYZPbJ@@[JDIaTwCH','sNp@DkfZj@[\\QXuqea`@','Hid@@DjU^nBAHAEVtaBXUMp`','dmt@@DjUgZDHJJ@C[PaLJfxYT','sGP@DkejPFsDVM@','sNx@@eJmUPB','qCb@AIf`H','HcL@@DjU_VnBBJh@FqaBXUMprnqcXX@','Hid@@DjUZnBBH@FtaBXUMpqahX@','sNp@LdbQZjBBvxbqkcGC@','sOx@@drm\\@@@|c@','sJP@H~j@^R@','KAx@@YIDcFjhDElHPfES\\','Hid@@DjUZnBAH@FtaBXUMp`','sNp@LddUji@[\\QXu`','sGP@DjfjPFsDVM@','HeT@@DjYUXPbD`@[JDIaTwCH','KAx@@IUoMUPAlHPfES]FDa`@','sFp@DiTt@@AyaD','Hed@@DjuuZjj@FraBXUMpsFIa`@','HeT@@DjUghP`h`@[JDIaTwCLXfF@','sOp@DjWkjj`FwDVM\\YhX@','sGP@Djfj`FsDVMcIC@','KAx@@IRkmUPHKXPaLJfzL]C@','sNx@@djmUPB','QM`AIdD','sOp@DjWkB@@Gbe@','sNp@DjyZj@[\\QXuqca`@','QM@HuAmd`','sNp@LddUjj@[\\QXuqea`@','daD@@DjUZxHD@CXPaLJfzLYC@','qCb@AIZPH','HiD@@LdbJTjjh@[RDIaTwCF@a`@','Hmt@@DjU_ZxHHi@AmhPfES\\Lj','HaDH@@RYWih@H@D','HiD@@LdbJTjjh@[RDIaTwCFHa`@','sGX@@djuT@|a@','sNp@DkfZj@[\\QXuqaa`@','Hid@@DjU^nBBH@GdL','KAx@@IVkMUPAlHPfES]FJa`@','qCr@XIKTAy@','HmT@@Dj{uVjjh@[ZDIaTwCJqaXX@','dmt@@DjYWVDHbJ@C[PaLJfxYT','Hif@@@RUe^Fh@@@P','HaDH@@Rfu[j@@@GdH','KAx@@IVsMUPAlHPfES]FDa`@','sKP@Di\\Zj@[TQXq`a`@','sJX@@eMU@OH@','HeT@@DjU^k``b`@[JDIaTwCLXFF@','Hmt@@LdbbRJXPbHh@FvaBXUMprh','sJP@DjvhAmQEcFBF@','dmt@@LdbbRNXPbHh@MmBDpj[aeXplL@','sJP`@dfvhAyD','sGP@Di^V`dmfHl[FVF@','KAx@@IVsmUPAlHPfES]FBa`@','sOq@@drm\\@@@|PP','sJY@BDeZhA@','HeT@@LdbRbmBDED@CYPaLJfxY@','Hed@@Djy[Zjj@FraBXUMpr','HeT@@DjU]k``b`@[JDIaTwCLXFF@','Hid@@DjUfaBB`@D','qCa@CIJtA@','QMPARVA@','Hid@@DjUfaBB`@FtaBXUMpqcHX@','sJY@BDfZhA@','HeT@@DjUghP`hP@[JDIaTwCH','Hed@@Dj{uZjj@FraBXUMpsFIa`@','dmt@@DjYUvDHbJ@C[PaLJfxYVLUC@','sNp`@dfuZj@P','sJQ@@dmU@OH@','sJX@@dmU@H','HeT@@DjU]k``b`@[JDIaTwCLXZF@','HiD@@LdfbJZjh@[RDIaTwCFAa`@','sOx@@drm\\@@@|a@','deT@@DjUfhP`h`@vTHSBinFP','dmt@@DjU]ZxHHj@C[PaLJfxYVLKC@','sOp@DjWkjj`FwDVM\\XHX@','HcL@@LdbbRNSBDQEP@McBDpj[ae]cFpp@','HiD@@Dj}Yji`AmHPfES\\H','HaDH@@RYe[hB@@D','Hid@@DjU^njjj@FtaBXUMpq`XX@','deT@@LdbbRKBDQB@FraBXUMpr','QMPARZA@','sOq@@drm\\@@@|QX','HaD@@DjYvxH`@A@','HcL@@LdbbRNcBDQEP@McBDpj[ae]@','QMhDRZA@','RG@DXLHmP','QM`BN`XQYd','RG@DTLHmP','QMHAIXFEVd','QMDARVAaH','RFPDXLHmP','RF@Q`vRbdLEC@','RF@QpvR@','QO@HyjAmd`','`II@B','`II@CFspqJp','`II@CF[@hM@prB`','`H@[T[|B`XN@PdM@p|@bHrBcDk@','RG@DXMj}F@','QM`BN`[L~b@','RG@DTMj}D@','QMHAIXFt~j@','QMDARVA}L@','RFPDXMj}D@','sKP@Di\\YZ@[TQXqaa`@','eF`BHD']);};Jrb.Fs=function Fs(a,b){var c,d;if(!b)return null;d=tX(Drb.NY,Qsb,6,(Jrb.Cs.length+31)/32|0,15,1);b=Jrb.Ks(b);Jrb.rs(a.g,b);for(c=0;c<Jrb.Cs.length;c++){Jrb.ps(a.g,Jrb.Ds[c]);Jrb.ds(a.g,1,4)>0&&(d[c/32|0]|=1<<31-c%32);}return d;};Jrb.Gs=function Gs(a,b){var c,d;if(!b)return null;d=tX(Drb.OY,iub,6,(Jrb.Cs.length+63)/64|0,14,1);b=Jrb.Ks(b);Jrb.rs(a.g,b);for(c=0;c<Jrb.Cs.length;c++){Jrb.ps(a.g,Jrb.Ds[c]);Jrb.ds(a.g,1,4)>0&&(d[c/64|0]=G3(d[c/64|0],H3(1,63-c%64)));}return d;};Jrb.Hs=function Hs(){var a,b;if(Jrb.Ds==null){b=new Jrb.Fp(false);Jrb.Ds=tX(Drb.JZ,mvb,27,Jrb.Cs.length,0,1);for(a=0;a<Jrb.Cs.length;a++){Jrb.Ds[a]=Jrb.tp(b,Jrb.Cs[a]);Jrb.ou(Jrb.Ds[a],1);}}};Jrb.Is=function Is(a){var b;if(a.f!=null){for(b=0;b<a.f.length;b++)if((a.c[b]&~a.f[b])!=0)return false;}else {return false;}return Jrb.Js(a);};Jrb.Js=function Js(a){!a.d&&(a.d=Jrb.vp(new Jrb.Fp(false),a.e));!a.a&&(a.a=Jrb.vp(new Jrb.Fp(false),a.b));Jrb.rs(a.g,a.d);Jrb.ps(a.g,a.a);return Jrb.ks(a.g);};Jrb.Ks=function Ks(a){var b,c;if(a.K){for(b=0;b<a.q;b++){if(E3(j3(a.B[b],jtb),0)){a=new Jrb.Iu(a);for(c=b;c<a.q;c++)E3(j3(a.B[c],jtb),0)&&(a.C[c]=-1);Jrb.gk(a);}}}return a;};Jrb.Ls=function Ls(a,b,c){a.b=null;a.a=b;c==null?a.c=Jrb.Fs(a,b):a.c=c;};Jrb.Ms=function Ms(a,b,c){a.e=null;a.d=b;c==null?a.f=Jrb.Fs(a,b):a.f=c;};Jrb.Ns=function Ns(){Jrb.Es();this.g=new Jrb.xs();Jrb.Hs();};Jrb.Os=function Os(a){Jrb.Es();var b,c,d,e,f,g;if(a==null)return null;b=tX(Drb.JY,nub,6,a.length*8,15,1);for(d=0;d<a.length;d++){g=a[d];for(e=7;e>=0;e--){c=g&15;c>9&&(c+=7);b[d*8+e]=48+c<<24>>24;g>>=4;}}return Erb.i7((f=b,hsb.Kab(),f));};Jrb.Ps=function Ps(a){Jrb.Es();var b,c,d,e;if(Hrb.rnb(a).length==0||(Hrb.rnb(a).length&7)!=0)return null;d=tX(Drb.NY,Qsb,6,Hrb.rnb(a).length/8|0,15,1);for(c=0;c<Hrb.rnb(a).length;c++){e=c/8|0;b=(Hrb.mnb(c,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(c)-48);b>16&&(b-=7);d[e]<<=4;d[e]+=b;}return d;};Jrb.Qs=function Qs(a,b){Jrb.Es();var c,d,e,f;f=0;d=0;e=0;for(c=0;c<a.length;c++){f+=p6(a[c]&b[c]);d+=p6(a[c]);e+=p6(b[c]);}return f/$wnd.Math.sqrt(d*e);};Jrb.Rs=function Rs(a,b){Jrb.Es();var c,d,e;e=0;c=0;for(d=0;d<a.length;d++){e+=p6(a[d]&b[d]);c+=p6(a[d]|b[d]);}return e/c;};Z3(141,1,{},Jrb.Ns);Drb.wZ=B5(141);Jrb.Ts=function Ts(a){a.j=1;a.k=10;a.n=400;a.f=400;a.i=true;a.d='black';a.b=new Krb.Yi();a.a=new Krb.Yi();a.c=new Erb.S7();a.e=new dsb.o4(12);};Jrb.Us=function Us(a,b){var c,d;c=Erb.U5(gsb.s8(gsb.o8(new gsb.t8(3*a.j),(gsb.Bab(),gsb.xab))));d='<line stroke-dasharray="'+c+','+c+btb+'x1="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.a),gsb.xab)))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.c),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.b),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b.d),gsb.xab)))+btb+ftb+a.d+btb+'stroke-width:'+a.j+htb;Jrb.dt(a,d);};Jrb.Vs=function Vs(a,b){var c,d;d=new Erb.U7('<polygon points="');for(c=0;c<b.a;c++){Erb.N7(d,u3($wnd.Math.round(b.b[c])));d.a+=',';Erb.N7(d,u3($wnd.Math.round(b.c[c])));d.a+=' ';}Erb.P7(d,'" fill="'+a.d+btb+ftb+a.d+btb+gtb+a.j+htb);Jrb.dt(a,d.a);};Jrb.Ws=function Ws(a,b,c,d){var e,f,g,h;g=(e=(h=dsb.m4(a.e,b),new esb.s4(h)).b,e);f='<text x="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c-g/2),(gsb.Bab(),gsb.xab))))+btb+'y="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(d+a.k/3),gsb.xab)))+btb+'stroke="none" '+'font-size="'+a.e.b+btb+'fill="'+a.d+'">'+b+'<\/text>';Jrb.dt(a,f);};Jrb.Xs=function Xs(a,b,c,d){var e;e='<circle cx="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(b+d/2),(gsb.Bab(),gsb.xab))))+btb+'cy="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c+d/2),gsb.xab)))+btb+'r="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(d/2),gsb.xab)))+btb+'fill="'+a.d+htb;Jrb.dt(a,e);};Jrb.Ys=function Ys(a,b,c,d){var e,f;e=Jrb.vk(a.K,b);f='<circle id="'+(a.g!=null?a.g:'mol'+Jrb.Ss)+':Atom:'+b+btb+nvb+(e==0?'':'data-atom-map-no="'+e+btb)+'cx="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c),(gsb.Bab(),gsb.xab))))+btb+'cy="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(d),gsb.xab)))+btb+'r="'+8+btb+ovb;Krb.Ei(a.a,f);};Jrb.Zs=function Zs(a,b,c,d,e,f){var g;g='<line id="'+(a.g!=null?a.g:'mol'+Jrb.Ss)+':Bond:'+b+btb+nvb+'x1="'+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(c),(gsb.Bab(),gsb.xab))))+btb+ctb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(d),gsb.xab)))+btb+dtb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(e),gsb.xab)))+btb+etb+Erb.U5(gsb.s8(gsb.o8(new gsb.t8(f),gsb.xab)))+btb+gtb+8+btb+ovb;Krb.Ei(a.b,g);};Jrb.$s=function $s(a,b){a.j=M3(u3($wnd.Math.round(100*$wnd.Math.max(b,1))))/100;};Jrb._s=function _s(a,b){a.d='rgb('+((b&dub)>>16)+','+((b&Ssb)>>8)+','+(b&255)+')';};Jrb.at=function at(a,b){if(a.k!=b){a.k=b;a.e=new dsb.o4(b);}};Jrb.bt=function bt(a,b){a.n=N3(u3($wnd.Math.round(b.b)));a.f=N3(u3($wnd.Math.round(b.a)));return Jrb.Rd(a,b);};Jrb.ct=function ct(a){var b,c,d,e,f,g;f='<svg id="'+(a.g!=null?a.g:'mol'+Jrb.Ss)+btb+'xmlns="http://www.w3.org/2000/svg" version="1.1" '+'width="'+a.n+'px" '+'height="'+a.f+'px" '+'viewBox="0 0 '+a.n+' '+a.f+'">\n';g=a.i?'<style> #'+(a.g!=null?a.g:'mol'+Jrb.Ss)+' { pointer-events:none; }'+' #'+(a.g!=null?a.g:'mol'+Jrb.Ss)+' .event '+' { pointer-events:all; }'+' line { stroke-linecap:round; }'+' polygon { stroke-linejoin:round; }'+' <\/style>\n':'<g style="font-size:'+a.k+'px; fill-opacity:1; stroke-opacity:1; fill:black; stroke:black;'+' font-weight:normal; text-rendering:optimizeLegibility; font-family:sans-serif;'+' stroke-linejoin:round; stroke-linecap:round; stroke-dashoffset:0;">';f+='  ';f+=g;if(a.i){for(e=new Krb.ueb(a.b);e.a<e.c.a.length;){d=Krb.teb(e);Jrb.dt(a,d);}for(c=new Krb.ueb(a.a);c.a<c.c.a.length;){b=Krb.teb(c);Jrb.dt(a,b);}}a.i||Jrb.dt(a,'<\/g>');return f+a.c.a+'<\/svg>';};Jrb.dt=function dt(a,b){Erb.P7(a.c,'  ');Erb.P7(a.c,b);Erb.P7(a.c,rsb);};Jrb.et=function et(a,b,c){Jrb.Vc();Jrb.Ud.call(this,a,b);Jrb.Ts(this);this.g=c;++Jrb.Ss;};Z3(228,263,{},Jrb.et);_.mb=function ft(){return Jrb.ct(this);};_.f=0;_.i=false;_.j=0;_.k=0;_.n=0;Jrb.Ss=0;Drb.yZ=B5(228);Jrb.gt=function gt(a,b){var c;if(a.c!=null)for(c=0;c<a.c.length;c++)if(b==a.c[c]&&a.d[c])return true;return false;};Jrb.ht=function ht(a,b,c,d,e){this.a=a;this.b=b;this.g=c;this.f=d;this.e=e;};Z3(150,1,{150:1},Jrb.ht);_.a=0;_.b=0;_.e=false;_.f=false;_.g=0;Drb.CZ=B5(150);Jrb.it=function it(a,b){!a.b&&(a.b=new Zrb.qV());Zrb.lV(a.b,Erb.A6(b));};Jrb.jt=function jt(a,b,c,d){var e,f,g,h,i,j;e=Jrb.Hj(b,a.d);Jrb.Sl(b,e,a.e);Jrb.$l(b,e,a.o,false);Jrb.Ql(b,e,a.a);if(E3(a.c,0)){if(E3(j3(a.c,2),0)){a.c=j3(a.c,-3);b.u[e]|=oub;}else {b.u[e]&=Jub;}Jrb.cm(b,e,a.c,true);}if(a.b){j=tX(Drb.NY,Qsb,6,a.b.d.a.length,15,1);for(i=0;i<a.b.d.a.length;i++)j[i]=Zrb.nV(a.b,i).a;Jrb.Yl(b,e,j);Hrb.Mmb(a.b.d.a,0);}else {if(String.fromCharCode(c).toLowerCase().charCodeAt(0)==c&&Erb.k5(c)){if(a.d!=5&&a.d!=6&&a.d!=7&&a.d!=8&&a.d!=15&&a.d!=16&&a.d!=33&&a.d!=34)throw h3(new Erb._z('SmilesParser: atomicNo '+a.d+' must not be aromatic. Position:'+(d-1)));b.u[e]|=oub;}else {b.u[e]&=Jub;}}if(a.f){for(h=new Krb.ueb(a.f);h.a<h.c.a.length;){g=Krb.teb(h);Jrb.jm(g,0,0);Jrb.Rj(b,g,e);}}if(a.g!=-1&&a.d!=1){f=tX(Drb.JY,nub,6,1,15,1);f[0]=a.g<<24>>24;Jrb.Wl(b,e,f);}return e;};Jrb.kt=function kt(a,b){var c;c=0;while(b<a.length&&(a[b]!=93||c!=0)){a[b]==91?++c:a[b]==93&&--c;++b;}if(b==a.length)throw h3(new Erb._z('SmilesParser: No matching closing bracket found.'));return b+1;};Jrb.lt=function lt(a,b,c){var d,e;if(b<3)return false;d=b-1;(a[d]==43||a[d]==45)&&--d;if(!i5(a[d]&Gsb))return false;--d;if(a[d]!=68&&a[d]!=82&&a[d]!=88&&a[d]!=122)return false;c[0]=0;while(d>0&&j5(a[d-1])){--d;++c[0];}e=b+1;while(j5(a[d])){if(a.length<=e||a[d]!==a[e])return false;++d;++e;}return true;};Jrb.mt=function mt(a,b,c,d,e,f){var g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v;if(b[c-1]==36){a.q=new Krb.Yi();c+=Jrb.qt(a,b,c-1,a.q)-1;if(b[c++]!=93){if(!f)throw h3(new Erb._z('SmilesParser: A positive recursive SMARTS followed by another one or by atom query features is not supported. Position:'+(c-1)));(a.k&128)==0&&(c=Jrb.kt(b,c));}return c;}if(b[c-1]==42){a.d=6;a.c=G3(a.c,1);}else if(b[c-1]==63){a.d=0;}else {j=b[c-1]==33;if(j){a.c=G3(a.c,1);++c;}if(b[c-1]==82&&e&&(i5(b[c]&Gsb)||a.j&&b[c]==123)){a.d=6;a.c=G3(a.c,1);--c;j&&--c;}else {k=new Jrb.st();if(!Jrb.nt(b,c-1,d,k))throw h3(new Erb._z("SmilesParser: Unexpected character in atom definition:'"+String.fromCharCode(b[c-1]&Gsb)+pvb+(c-1)));a.d=k.a;c+=k.b-1;(a.k&3)!=2&&(a.g=9);if(e&&(b[c]==44||j)){n=k.d;m=k.c;t=c-k.b;while(t<d){if(!Jrb.nt(b,t,d,k)){if(!j)throw h3(new Erb._z("SmilesParser: Unexpected character in atom list:'"+String.fromCharCode(b[t]&Gsb)+qvb+t));break;}if(k.a==1){if(!j)throw h3(new Erb._z("SmilesParser: Hydrogen is not supported in positive atom lists:'"+Erb.h7((Krb.zeb(b,t,d),Krb.Eeb(b,tX(Drb.JY,nub,6,d-t,15,1),t,d)))+qvb+t));}else {Jrb.it(a,k.a);n=n|k.d;m=m|k.c;}t+=k.b;if(j&&b[t]!=59&&b[t]!=38)break;if(!j&&b[t]!=44)break;if(j&&b[t+1]!=33)break;++t;b[t]==33&&++t;}if(!!a.b&&a.b.d.a.length>1){a.g=-1;m?n||(a.c=G3(a.c,4)):a.c=G3(a.c,2);}c=t;}}}p=new Jrb.wt(b);r=tX(Drb.NY,Qsb,6,1,15,1);s=true;while(s){if(b[c]==64){++c;if(b[c]==64){a.i=true;++c;}a.p=true;continue;}if(b[c]==58){++c;while(i5(b[c]&Gsb)){a.o=10*a.o+b[c]-48;++c;}continue;}if(b[c]==91)throw h3(new Erb._z('SmilesParser: nested square brackets found. Position:'+c));if(b[c]==93){++c;s=false;continue;}a.e=Jrb.pt(b,c,r);if(r[0]!=0){c+=r[0];a.e==0&&(a.c=G3(a.c,xtb));continue;}i=b[c]==33;i&&++c;if(b[c]==72){++c;c+=Jrb.tt(p,c,1);g=0;p.c<=0&&p.b>=0&&(g=G3(g,128));p.c<=1&&p.b>=1&&(g=G3(g,256));p.c<=2&&p.b>=2&&(g=G3(g,512));p.c<=3&&p.b>=3&&(g=G3(g,vtb));if(i){a.c=G3(a.c,g);a.g=-1;}else {if(p.b==p.c){a.g=p.c;}else {a.c=G3(a.c,j3(utb,F3(g)));a.g=-1;}}continue;}if(b[c]==68||b[c]==100){++c;c+=Jrb.tt(p,c,1);g=0;p.c<=0&&p.b>=0&&(g=G3(g,oub));p.c<=1&&p.b>=1&&(g=G3(g,bub));p.c<=2&&p.b>=2&&(g=G3(g,aub));p.c<=3&&p.b>=3&&(g=G3(g,qub));p.c<=4&&p.b>=4&&(g=G3(g,Etb));if(o3(g,0)!=0){if(i)a.c=G3(a.c,g);else if(E3(j3(a.c,Btb),0))a.c=j3(a.c,F3(g));else {g=P3(g,Btb);a.c=G3(a.c,g);}}continue;}if(b[c]==122&&a.j){++c;c+=Jrb.tt(p,c,4);g=0;p.c<=0&&p.b>=0&&(g=G3(g,Otb));p.c<=1&&p.b>=1&&(g=G3(g,rvb));p.c<=2&&p.b>=2&&(g=G3(g,svb));p.c<=3&&p.b>=3&&(g=G3(g,tvb));p.c<=4&&p.b>=4&&(g=G3(g,Ntb));if(o3(g,0)!=0){if(i)a.c=G3(a.c,g);else if(E3(j3(a.c,Itb),0))a.c=j3(a.c,F3(g));else {g=P3(g,Itb);a.c=G3(a.c,g);}}continue;}if(b[c]==88){++c;c+=Jrb.tt(p,c,1);v=(Jrb.Ej(),Jrb.zj)[a.d];if(v==null)continue;u=v[0];l=Jrb.pt(b,c,r);r[0]!=0&&(Jrb.Ym(a.d)?u+=l:a.d==6?u-=$wnd.Math.abs(l):u-=l);g=0;u-p.c<=0&&u-p.b>=0&&(g=G3(g,ptb));u-p.c<=1&&u-p.b>=1&&(g=G3(g,Iub));u-p.c<=2&&u-p.b>=2&&(g=G3(g,Fsb));if(o3(g,0)!=0){if(i)a.c=G3(a.c,g);else if(E3(j3(a.c,Atb),0))a.c=j3(a.c,F3(g));else {g=P3(g,Atb);a.c=G3(a.c,g);}}continue;}if(b[c]==65||b[c]==97){a.c=G3(a.c,i^b[c]==65?4:2);++c;continue;}if(b[c]==82){++c;c+=Jrb.tt(p,c,3);g=0;p.c<=0&&p.b>=0&&(g=G3(g,8));p.c<=1&&p.b>=1&&(g=G3(g,16));p.c<=2&&p.b>=2&&(g=G3(g,32));p.c<=3&&p.b>=3&&(g=G3(g,64));p.b>3&&Jrb.Ut(a.n,(i?'!R':'R')+p.b);if(o3(g,0)!=0){if(i)a.c=G3(a.c,g);else if(E3(j3(a.c,120),0))a.c=j3(a.c,F3(g));else {g=P3(g,120);a.c=G3(a.c,g);}}continue;}if(b[c]==114){++c;c+=Jrb.tt(p,c,1);if(p.a){i?a.c=G3(a.c,384):a.c=G3(a.c,8);continue;}q=p.c;p.b>p.c&&Jrb.Ut(a.n,(i?'!r':'r')+('{'+p.c+'-'+p.b+'}'));!i&&q>=3&&q<=7?a.c=G3(a.c,Drb.r3(q<<22)):p.b>p.c||Jrb.Ut(a.n,(i?'!r':'r')+q);continue;}if(b[c]==118){++c;c+=Jrb.tt(p,c,1);u=p.c;p.b>p.c&&Jrb.Ut(a.n,(i?'!v':'v')+('{'+p.c+'-'+p.b+'}'));!i&&u<=14?a.a=u:p.b>p.c||Jrb.Ut(a.n,(i?'!v':'v')+u);continue;}if(b[c]==94){++c;h=b[c++]-48;if(h<1||h>3)throw h3(new Erb._z('SmilesParser: Unsupported hybridization. Position:'+c));o=h==1?Fsb:h==2?Iub:ptb;i||(o=j3(Atb,F3(o)));a.c=G3(a.c,o);continue;}if(b[c]==36){if(!i)throw h3(new Erb._z('SmilesParser: non-negated recursive SMARTS relating to preceding atom are not supported yet. Position:'+c));!a.f&&(a.f=new Krb.Yi());c+=Jrb.qt(a,b,c,a.f);continue;}if(e&&(b[c]==59||b[c]==38)){++c;continue;}if(e&&b[c]==44&&Jrb.lt(b,c,r)){c+=r[0]+1;continue;}if(e&&b[c]==44&&(a.k&128)!=0){c+=1;break;}if(b[c]==44)throw h3(new Erb._z("SmilesParser: alternative atom definitions not supported. (Tip: enumerate SMARTS): '"+String.fromCharCode(b[c]&Gsb)+uvb+c));throw h3(new Erb._z("SmilesParser: unexpected character inside brackets: '"+String.fromCharCode(b[c]&Gsb)+uvb+c));}return c;};Jrb.nt=function nt(a,b,c,d){var e,f;d.d=true;d.c=true;if(a[b]==35){++b;d.a=0;d.b=1;while(b<c&&i5(a[b]&Gsb)){d.a=10*d.a+a[b]-48;++d.b;++b;}if(d.a==0||d.a>=(Jrb.Ej(),Jrb.yj).length)throw h3(new Erb._z('SmilesParser: Atomic number out of range. position:'+(b-1)));return true;}if(a[b]>=65&&a[b]<=90){d.b=a[b+1]>=97&&a[b+1]<=122?2:1;d.a=(Jrb.Ej(),Jrb.Xm(Erb.D7(Hrb.Smb(a,b,(f=d.b,hsb.Kab(),f))),321));if(d.b==2&&d.a==0){d.b=1;d.a=Jrb.Xm(Erb.D7(Hrb.Smb(a,b,(e=d.b,e))),321);}d.d=false;if(d.a==0)throw h3(new Erb._z('SmilesParser: Unknown atom label. position:'+(b-1)));return true;}if(a[b]==65&&a[b+1]==115||a[b]==83&&a[b+1]==101){d.b=2;d.a=(Jrb.Ej(),Jrb.Xm(Erb.D7(Hrb.Smb(a,b,(e=d.b,hsb.Kab(),e))),321));d.c=false;return true;}if(a[b]==99||a[b]==110||a[b]==111||a[b]==112||a[b]==115){d.b=1;d.a=(Jrb.Ej(),Jrb.Xm(Erb.D7(Hrb.Smb(a,b,(e=d.b,hsb.Kab(),e))),321));d.c=false;return true;}return false;};Jrb.ot=function ot(a,b,c,d,e){if(b[c-1]==42){a.d=6;a.c=G3(a.c,1);}else if(b[c-1]==63){a.d=0;}else if((b[c-1]==65||b[c-1]==97)&&e){a.d=6;a.c=G3(a.c,1);a.c=G3(a.c,b[c-1]==65?4:2);}else {switch(String.fromCharCode(b[c-1]).toUpperCase().charCodeAt(0)){case 66:if(c<d&&b[c]==114){a.d=35;++c;}else a.d=5;break;case 67:if(c<d&&b[c]==108){a.d=17;++c;}else a.d=6;break;case 70:a.d=9;break;case 73:a.d=53;break;case 78:a.d=7;break;case 79:a.d=8;break;case 80:a.d=15;break;case 83:a.d=16;}}return c;};Jrb.pt=function pt(a,b,c){var d,e;c[0]=0;if(a[b]==43||a[b]==45){e=a[b];d=1;++c[0];while(a[b+c[0]]==e){++d;++c[0];}if(d==1&&i5(a[b+1]&Gsb)){d=a[b+1]-48;++c[0];}return e==43?d:-d;}return 0;};Jrb.qt=function qt(a,b,c,d){var e,f,g,h,i;if(b[c+1]!=40)throw h3(new Erb._z("SmilesParser: '$' for recursive SMARTS must be followed by '('. position:"+c));h=1;f=c+2;while(f<b.length&&h>0){b[f]==40?++h:b[f]==41&&--h;++f;}if(h>0)throw h3(new Erb._z("SmilesParser: Missing closing ')' for recursive SMARTS. '('-position:"+(c+1)));g=new Jrb.Hu(16,16);Jrb.vm(g,true);i=new Jrb.Wt(a.k);Jrb.St(i,a.n.f);Jrb.Kt(i,g,b,c+2,f-1,true,true);Hrb.Kmb(d.a,g);if(b[c-1]==33)for(e=0;e<g.q;e++)Jrb.cm(g,e,jtb,true);return f-c;};Jrb.rt=function rt(a,b){this.n=a;this.k=b;this.j=(b&16)==0;this.d=-1;this.e=0;this.o=0;this.a=-1;this.g=-1;this.c=0;};Z3(248,1,{},Jrb.rt);_.a=0;_.c=0;_.d=0;_.e=0;_.g=0;_.i=false;_.j=false;_.k=0;_.o=0;_.p=false;Drb.BZ=B5(248);Jrb.st=function st(){this.a=-1;};Z3(312,1,{},Jrb.st);_.a=0;_.b=0;_.c=false;_.d=false;Drb.zZ=B5(312);Jrb.tt=function tt(a,b,c){var d,e,f,g,h;a.a=false;a.d=b;if(i5(a.e[b]&Gsb)){h=Jrb.ut(a);a.c=a.b=h;d=b-1;while(d>1&&l5(a.e[d-1]))--d;while(a.e[a.d]==44){g=true;f=b-d;for(e=0;e<f;e++){if(a.e[d+e]!==a.e[a.d+1+e]){g=false;break;}}if(!g)break;a.d+=1+f;h=Jrb.ut(a);a.c>h?a.c=h:a.b<h&&(a.b=h);}return a.d-b;}if(a.e[b]==123&&i5(a.e[b+1]&Gsb)){++a.d;a.c=Jrb.ut(a);if(a.e[a.d++]!=45)return 0;if(!i5(a.e[a.d]&Gsb))return 0;a.b=Jrb.ut(a);if(a.e[a.d++]!=125)return 0;return a.d-b;}a.c=1;a.b=c;a.a=true;return 0;};Jrb.ut=function ut(a){var b;b=a.e[a.d++]-48;i5(a.e[a.d]&Gsb)&&(b=10*b+(a.e[a.d++]-48));return b;};Jrb.vt=function vt(a){return '{'+a.c+'-'+a.b+'}';};Jrb.wt=function wt(a){this.e=a;};Z3(313,1,{},Jrb.wt);_.mb=function xt(){return Jrb.vt(this);};_.a=false;_.b=0;_.c=0;_.d=0;Drb.AZ=B5(313);Jrb.yt=function yt(a,b){var c,d,e,f;f=false;a.b=b;Jrb.ou(a.b,15);c=a.b.f;d=a.b.g;a.j=tX(Drb.c3,itb,6,d,16,1);for(e=0;e<d;++e)a.j[e]=false;a.g=tX(Drb.c3,itb,6,c,16,1);a.c=tX(Drb.NY,Qsb,6,c,15,1);for(e=0;e<c;++e){a.g[e]=false;a.c[e]=-1;}a.e=tX(Drb.B0,Esb,2,3*c,6,1);a.i=0;a.d=0;a.a=0;while(!f){for(e=0;e<c;++e){if(!a.g[e]){a.a>0&&(a.e[a.i++]='.');Jrb.At(a,e,-1);++a.a;break;}}e==c&&(f=true);}a.f='';for(e=0;e<a.i;++e)a.f+=''+a.e[e];return a.f;};Jrb.zt=function zt(a){switch(a){case 5:case 6:case 7:case 8:case 9:case 15:case 16:case 17:case 35:case 53:return true;default:return false;}};Jrb.At=function At(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;d=true;i=0;p=0;m=a.i;a.c[b]=m;h=Jrb.Dk(a.b,b);g=Jrb.sk(a.b,b);e=Jrb.mk(a.b,b);f=Jrb.wk(a.b,b);k=Jrb.Gn(a.b,b);e==0&&f==0&&Jrb.zt(h)&&(d=false);a.e[m]='';if(c!=-1){switch(Jrb.Qk(a.b,c)){case 0:a.e[m]+='~';break;case 2:a.e[m]+='=';break;case 3:a.e[m]+='#';}}d&&(a.e[m]+='[');f!=0&&(a.e[m]+=''+f);a.e[m]+=''+g;if(d){if(0<(o=Jrb.Wn(a.b,b))){a.e[m]+='H';1<o&&(a.e[m]+=o);}}if(e!=0){e>0?a.e[m]+='+':a.e[m]+='-';$wnd.Math.abs(e)>1&&(a.e[m]+=''+$wnd.Math.abs(e));}d&&(a.e[m]+=']');c!=-1&&(a.j[c]=true);a.g[b]=true;++a.i;for(n=0;n<k;++n)a.j[Jrb.Hn(a.b,b,n)]||++i;for(n=0;n<k;++n){j=Jrb.Fn(a.b,b,n);l=Jrb.Hn(a.b,b,n);if(a.j[l]){++p;continue;}if(a.g[j]){++a.d;a.j[l]=true;switch(Jrb.Qk(a.b,l)){case 0:a.e[a.c[j]]+='~';a.e[m]+='~';break;case 2:a.e[a.c[j]]+='=';a.e[m]+='=';break;case 3:a.e[a.c[j]]+='#';a.e[m]+='3';}if(a.d>9){a.e[a.c[j]]+='%';a.e[m]+='%';}a.e[a.c[j]]+=''+a.d;a.e[m]+=''+a.d;continue;}n-p<i-1&&(a.e[a.i++]='(');Jrb.At(a,j,l);n-p<i-1&&(a.e[a.i++]=')');}};Jrb.Bt=function Bt(){};Z3(227,1,{},Jrb.Bt);_.a=0;_.d=0;_.i=0;Drb.DZ=B5(227);Jrb.Ct=function Ct(a,b,c,d,e,f){var g;g=Jrb.Jj(a.k,b,c,d);Krb.xbb(a.s,b+'_'+c,BX(nX(Drb.NY,1),Qsb,6,15,[e,f]));Krb.xbb(a.s,c+'_'+b,BX(nX(Drb.NY,1),Qsb,6,15,[e,f]));Krb.tfb(a.a,b);Krb.tfb(a.a,c);return g;};Jrb.Dt=function Dt(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;j=tX(Drb.NY,Qsb,6,a.k.f,15,1);h=tX(Drb.NY,Qsb,6,a.k.f,15,1);i=tX(Drb.NY,Qsb,6,a.k.f,15,1);k=tX(Drb.NY,Qsb,6,a.k.f,15,1);c=Jrb.Ik(a.k,0,b);d=Jrb.Ik(a.k,1,b);h[0]=c;h[1]=d;i[0]=-1;i[1]=b;j[c]=1;j[d]=2;k[c]=-1;k[d]=c;g=1;l=1;while(g<=l&&j[h[g]]<15){o=h[g];for(m=0;m<Jrb.Gn(a.k,o);m++){e=Jrb.Fn(a.k,o,m);if(e!=k[o]){f=Jrb.Hn(a.k,o,m);if(e==c){i[0]=f;for(n=0;n<=l;n++){if(!a.g[i[m]]){a.g[i[m]]=true;++a.c;}}return;}if(Jrb.sl(a.k,e)&&j[e]==0){++l;h[l]=e;i[l]=f;j[e]=j[o]+1;k[e]=o;}}}++g;}return;};Jrb.Et=function Et(a){var b,c,d,e,f,g,h,i,j,k,l,m,n;Jrb.ou(a.k,7);l=false;m=tX(Drb.NY,Qsb,6,2,15,1);n=tX(Drb.NY,Qsb,6,2,15,1);k=tX(Drb.NY,Qsb,6,2,15,1);for(d=0;d<a.k.g;d++){if(!Jrb.Po(a.k,d)&&Jrb.Tk(a.k,d)==2){for(g=0;g<2;g++){m[g]=-1;k[g]=-1;b=Jrb.Ik(a.k,g,d);for(j=0;j<Jrb.Gn(a.k,b);j++){e=Jrb.Hn(a.k,b,j);if(e!=d){if(m[g]==-1&&(Jrb.Tk(a.k,e)==257||Jrb.Tk(a.k,e)==129)){m[g]=Jrb.Fn(a.k,b,j);n[g]=e;}else {k[g]=Jrb.Fn(a.k,b,j);}}}if(m[g]==-1)break;}if(m[0]!=-1&&m[1]!=-1){i=Jrb.Tk(a.k,n[0])==Jrb.Tk(a.k,n[1]);for(h=0;h<2;h++)m[h]==Jrb.Ik(a.k,0,n[h])&&(i=!i);for(f=0;f<2;f++)k[f]!=-1&&k[f]<m[f]&&(i=!i);Jrb.qm(a.k,d,i?2:1,false);l=true;}}}for(c=0;c<a.k.g;c++)(Jrb.Tk(a.k,c)==257||Jrb.Tk(a.k,c)==129)&&Jrb.tm(a.k,c,1);return l;};Jrb.Ft=function Ft(a){return a==61?2:a==35?4:a==36?32:a==58?8:a==62?16:a==126?31:1;};Jrb.Gt=function Gt(a,b){var c,d,e,f,g,h,i,j,k,l,m;for(c=0;c<a.k.f;c++){if(Jrb.sl(a.k,c)){i=tX(Drb.NY,Qsb,6,a.k.f,15,1);h=tX(Drb.NY,Qsb,6,a.k.f,15,1);j=tX(Drb.NY,Qsb,6,a.k.f,15,1);h[0]=c;i[c]=1;j[c]=-1;g=0;k=0;while(g<=k){e=(i[h[g]]&1)==1?1:2;for(l=0;l<Jrb.Gn(a.k,h[g]);l++){d=Jrb.Hn(a.k,h[g],l);if(Jrb.Qk(a.k,d)==e&&b[d]){f=Jrb.Fn(a.k,h[g],l);if(i[f]==0){if(e==1&&Jrb.sl(a.k,f)){m=h[g];while(m!=-1){Jrb.tm(a.k,Jrb.Dn(a.k,f,m),e==1?2:1);e=3-e;f=m;m=j[m];}Jrb._l(a.k,c,false);Jrb._l(a.k,f,false);a.b-=2;return true;}h[++k]=f;j[f]=h[g];i[f]=i[h[g]]+1;}}}++g;}}}return false;};Jrb.Ht=function Ht(a){var b,c,d,e;for(b=0;b<a.k.f;b++){if(Jrb.Dk(a.k,b)==7&&Jrb.mk(a.k,b)==0&&Jrb.ao(a.k,b)>3&&Jrb.wn(a.k,b)>0){for(e=0;e<Jrb.Gn(a.k,b);e++){c=Jrb.Fn(a.k,b,e);d=Jrb.Hn(a.k,b,e);if(Jrb.Qk(a.k,d)>1&&Jrb.ql(a.k,c)){Jrb.Tk(a.k,d)==4?Jrb.tm(a.k,d,2):Jrb.tm(a.k,d,1);Jrb.Sl(a.k,b,Jrb.mk(a.k,b)+1);Jrb.Sl(a.k,c,Jrb.mk(a.k,c)-1);Jrb.Ql(a.k,b,-1);break;}}}}};Jrb.It=function It(a){return !a.r?'':'Unresolved SMARTS features:'+a.r;};Jrb.Jt=function Jt(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U;Jrb.ou(a.k,1);a.g=tX(Drb.c3,itb,6,a.k.g,16,1);a.c=0;for(l=0;l<a.k.g;l++){if(Jrb.Tk(a.k,l)==8){Jrb.tm(a.k,l,1);a.g[l]=true;++a.c;}}t=tX(Drb.c3,itb,6,a.k.f,16,1);U=new Jrb.Tr(a.k,3);s=tX(Drb.c3,itb,6,U.i.a.length,16,1);for(Q=0;Q<U.i.a.length;Q++){S=Krb.Ji(U.i,Q);s[Q]=true;for(A=S,D=0,H=A.length;D<H;++D){v=A[D];if(!Jrb.sl(a.k,v)){s[Q]=false;break;}}if(s[Q]){for(B=S,F=0,I=B.length;F<I;++F){v=B[F];t[v]=true;}T=Krb.Ji(U.j,Q);for(w=T,C=0,G=w.length;C<G;++C){v=w[C];if(!a.g[v]){a.g[v]=true;++a.c;}}}}for(m=0;m<a.k.g;m++){!a.g[m]&&(U.b[m]&Gsb)!=0&&Jrb.sl(a.k,Jrb.Ik(a.k,0,m))&&Jrb.sl(a.k,Jrb.Ik(a.k,1,m))&&Jrb.Dt(a,m);}for(n=0;n<a.k.g;n++){if(!a.g[n]){f=Jrb.Ik(a.k,0,n);g=Jrb.Ik(a.k,1,n);if(!t[f]&&!t[g]&&Jrb.sl(a.k,f)&&Jrb.sl(a.k,g)){a.g[n]=true;++a.c;}}}Jrb.ou(a.k,7);(a.q==2||a.q==1&&a.p)&&Jrb.Qt(a);r=tX(Drb.c3,itb,6,a.k.g,16,1);a.k.g>=0&&Erb.Y7(a.g,0,r,0,a.k.g);for(R=0;R<U.i.a.length;R++){if(s[R]){S=Krb.Ji(U.i,R);for(K=S,L=0,M=K.length;L<M;++L){J=K[L];if(!Jrb.Rt(a,J)){if(Jrb.sl(a.k,J)){Jrb._l(a.k,J,false);--a.b;}for(v=0;v<Jrb.Gn(a.k,J);v++){p=Jrb.Hn(a.k,J,v);if(a.g[p]){a.g[p]=false;--a.c;}}}}}}Jrb.Pt(a);for(P=0;P<U.i.a.length;P++){if(s[P]&&Krb.Ji(U.j,P).length==6){T=Krb.Ji(U.j,P);u=true;for(i=T,j=0,k=i.length;j<k;++j){h=i[j];if(!a.g[h]){u=false;break;}}if(u){Jrb.Ot(a,T[0]);Jrb.Ot(a,T[2]);Jrb.Ot(a,T[4]);Jrb.Pt(a);}}}for(O=5;O>=4;O--){do{N=false;for(h=0;h<a.k.g;h++){if(a.g[h]){d=0;for(q=0;q<2;q++){o=Jrb.Ik(a.k,q,h);for(v=0;v<Jrb.Gn(a.k,o);v++)a.g[Jrb.Hn(a.k,o,v)]&&++d;}if(d==O){Jrb.Ot(a,h);Jrb.Pt(a);N=true;break;}}}}while(N);}while(a.b>=2)if(!Jrb.Gt(a,r))break;if(b){if(a.b!=0){for(e=0;e<a.k.f;e++){if(Jrb.sl(a.k,e)){Jrb._l(a.k,e,false);Jrb.cm(a.k,e,2,true);--a.b;c=true;}}}if(a.c!=0){for(h=0;h<a.k.g;h++){if(a.g[h]){a.g[h]=false;Jrb.tm(a.k,h,8);--a.c;c=true;}}}}else {for(e=0;e<a.k.f;e++){if(Jrb.sl(a.k,e)&&Jrb.Wn(a.k,e)!=0){Jrb._l(a.k,e,false);Jrb.dm(a.k,e,32);--a.b;}}}if(a.q==0||a.q==1&&!c){if(a.b!=0)throw h3(new Erb._z(vvb));if(a.c!=0)throw h3(new Erb._z(vvb));}};Jrb.Kt=function Kt(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$,ab,bb,cb,db,eb,fb,gb,hb,ib;a.k=b;Jrb.mu(a.k);a.s=new Krb.thb();a.a=new Krb.ufb();!!a.r&&Erb.G4(a.r,0);a.b=0;a.p=false;j=a.q!=0;T=null;o=tX(Drb.NY,Qsb,6,32,15,1);o[0]=-1;Y=tX(Drb.NY,Qsb,6,16,15,1);ab=tX(Drb.NY,Qsb,6,16,15,1);$=tX(Drb.NY,Qsb,6,16,15,1);Z=tX(Drb.NY,Qsb,6,16,15,1);for(J=0;J<16;J++)Y[J]=-1;m=0;C=-1;bb=false;L=false;G=false;t=0;r=1;q=0;while(c[d]<=32)++d;while(d<e){cb=c[d++]&Gsb;if(Erb.k5(cb)||cb==42||cb==63||cb==33&&j&&bb||cb==35&&j&&bb||cb==36&&j&&bb){n=new Jrb.rt(a,a.j|a.q);if(bb){if((a.j&128)!=0){w=new Jrb.Yt(d-1);d=Jrb.mt(n,c,d,e,true,true);if(c[d-1]!=93){while(c[d-1]!=93){d=Jrb.mt(n,c,d+1,e,true,true);++w.a;}a.f.add(w);}}else {d=Jrb.mt(n,c,d,e,j,false);}}else {d=Jrb.ot(n,c,d,e,j);}bb=false;if(!n.q?null:Krb.Ji(n.q,0)){C=o[t];o[t]=b.q;Jrb.Kj(b,!n.q?null:Krb.Ji(n.q,0));if(C!=-1&&r!=512){p=Jrb.Jj(a.k,C,C,r);if(q!=0){a.p=true;Jrb.sm(a.k,p,q,true);Jrb.Fj(a.k,p);}}r=1;q=0;continue;}if(n.d==-1&&cb!=63)throw h3(new Erb._z('SmilesParser: unknown element label found. Position:'+(d-1)));(E3(n.c,0)||!!n.b)&&(a.p=true);l=Jrb.jt(n,a.k,cb,d);Jrb.sl(a.k,l)&&++a.b;C=o[t];if(C!=-1&&r!=512){p=Jrb.Jj(a.k,C,l,r);if(q!=0){a.p=true;Jrb.sm(a.k,p,q,true);Jrb.Fj(a.k,p);}}r=1;q=0;o[t]=l;if(m!=0){Jrb.am(a.k,l,m);m=0;}if(g){R=!T?null:Krb.kdb(T,Erb.A6(C));!!R&&Jrb.$t(R,l,(M=d,n.d==1&&Jrb.wk(a.k,l)==0,M));if(n.p){!T&&(T=new Krb.xjb());H=n.g==9?0:n.g;Krb.pjb(T,Erb.A6(l),new Jrb.eu(a,l,d-2,C,H,n.i));}}continue;}if(cb==46){o[t]=-1;r=512;continue;}if(cb==45||cb==61||cb==35||cb==36||cb==58||cb==47||cb==92||cb==60||cb==126||cb==33||cb==64){if(bb)throw h3(new Erb._z("SmilesParser: unexpected bond symbol inside square brackets: '"+String.fromCharCode(cb)+uvb+(d-1)));A=0;while(cb==45||cb==61||cb==35||cb==36||cb==58||cb==47||cb==92||cb==60||cb==126||cb==33||cb==64){if(cb==33){cb=c[d++]&Gsb;if(cb==64)q|=128;else if(cb==45&&c[d]==62||cb==60&&c[d]==45){A|=16;++d;}else if(cb==45)A|=1;else if(cb==61)A|=2;else if(cb==35)A|=4;else if(cb==36)A|=32;else if(cb==58)A|=8;else throw h3(new Erb._z("SmilesParser: bond symbol '"+String.fromCharCode(cb)+"' not allowed after '!'. Position:"+(d-1)));}else {if(cb==64)q|=256;else if(cb==61)r=2;else if(cb==35)r=4;else if(cb==36)r=32;else if(cb==58)r=8;else if(cb==126)q|=15;else if(cb==47){g&&(r=257);}else if(cb==92){g&&(r=129);}else if(cb==45&&c[d]==62||cb==60&&c[d]==45){r=16;++d;}if(c[d]==44){q|=(r==16?62:cb)==61?2:(r==16?62:cb)==35?4:(r==16?62:cb)==36?32:(r==16?62:cb)==58?8:(r==16?62:cb)==62?16:(r==16?62:cb)==126?31:1;while(c[d]==44){if(c[d+1]==60&&c[d+2]==45||c[d+1]==45&&c[d+2]==62){q|=16;d+=3;}else {q|=Jrb.Ft(c[d+1]&Gsb);d+=2;}}}}if(c[d]==59){++d;cb=c[d++]&Gsb;continue;}A!=0&&(q|=31&~A);break;}continue;}if(cb<=32){d=e;continue;}if(i5(cb)){O=cb-48;if(bb){while(d<e&&i5(c[d]&Gsb)){O=10*O+c[d]-48;++d;}m=O;}else {s=L?d-3:d-2;F=c[s]==45||c[s]==47||c[s]==92||c[s]==61||c[s]==35||c[s]==36||c[s]==58||c[s]==62||c[s]==126;if(L&&d<e&&i5(c[d]&Gsb)){O=10*O+c[d]-48;L=false;++d;}if(O>=Y.length){if(O>=100)throw h3(new Erb._z('SmilesParser: ringClosureAtom number out of range: '+O));P=Y.length;N=Y.length;while(N<=O)N=$wnd.Math.min(100,N+16);Y=(Hrb.cnb(N),Krb.Eeb(Y,tX(Drb.NY,Qsb,6,N,15,1),0,N));ab=(Hrb.cnb(N),Krb.Eeb(ab,tX(Drb.NY,Qsb,6,N,15,1),0,N));$=(Hrb.cnb(N),Krb.Eeb($,tX(Drb.NY,Qsb,6,N,15,1),0,N));Z=(Hrb.cnb(N),Krb.Eeb(Z,tX(Drb.NY,Qsb,6,N,15,1),0,N));for(I=P;I<N;I++)Y[I]=-1;}if(Y[O]==-1){Y[O]=o[t];ab[O]=d-1;$[O]=F?r:-1;Z[O]=F?q:0;}else {if(Y[O]==o[t])throw h3(new Erb._z('SmilesParser: ring closure to same atom'));if(g&&!!T){R=Krb.kdb(T,Erb.A6(Y[O]));!!R&&Jrb.$t(R,o[t],ab[O]);R=Krb.kdb(T,Erb.A6(o[t]));!!R&&Jrb.$t(R,Y[O],d-1);}$[O]!=-1?r=$[O]:r==257?r=129:r==129&&(r=257);h=Y[O];i=o[t];p=Jrb.Ct(a,h,i,r,ab[O],d-1);Z[O]!=0&&(q=Z[O]);if(q!=0){a.p=true;Jrb.sm(a.k,p,Z[O],true);Jrb.Fj(a.k,p);}Y[O]=-1;}r=1;q=0;}continue;}if(cb==43){throw h3(new Erb._z("SmilesParser: '+' found outside brackets. Position:"+(d-1)));}if(cb==40){if(o[t]==-1){G=true;continue;}++t;o.length==t&&(o=Krb.Ceb(o,o.length+32));o[t]=o[t-1];continue;}if(cb==41){if(t==0){if(!G)throw h3(new Erb._z("SmilesParser: Closing ')' without opening counterpart. Position:"+(d-1)));o[0]=-1;G=false;continue;}--t;continue;}if(cb==91){bb=true;continue;}if(cb==93){throw h3(new Erb._z('SmilesParser: closing bracket at unexpected position:'+(d-1)));}if(cb==37){L=true;continue;}throw h3(new Erb._z("SmilesParser: unexpected character outside brackets: '"+String.fromCharCode(cb)+pvb+(d-1)));}if(r!=1)throw h3(new Erb._z('SmilesParser: dangling open bond'));for(V=Y,W=0,X=V.length;W<X;++W){U=V[W];if(U!=-1)throw h3(new Erb._z('SmilesParser: dangling ring closure.'));}D=Jrb.Tn(a.k);Jrb.xm(a.k,true);Jrb.ou(a.k,1);for(k=0;k<a.k.q;k++){if(Jrb.ok(a.k,k)!=null){B=Jrb.pk(a.k,k)[0];if(a.p||a.q==2){if(B!=9){if(a.i){for(I=0;I<B;I++)Jrb.Jj(a.k,k,Jrb.Hj(a.k,1),1);}else {B==0&&Jrb.cm(a.k,k,1792,true);B==1&&Jrb.cm(a.k,k,1664,true);B==2&&Jrb.cm(a.k,k,1408,true);B==3&&Jrb.cm(a.k,k,896,true);}}}else {B==9&&(B=0);if(!Jrb.tl(a.k,k)&&(!Jrb.sl(a.k,k)||Jrb.Dk(a.k,k)==6&&Jrb.mk(a.k,k)==0)){ib=Jrb.Um(Jrb.Dk(a.k,k));u=false;db=Jrb.ao(a.k,k);db-=Jrb.Xk(a.k,k,db);db+=B;Jrb.sl(a.k,k)&&++db;for(fb=ib,gb=0,hb=fb.length;gb<hb;++gb){eb=fb[gb];if(db<=eb){u=true;eb==db+2?Jrb.dm(a.k,k,48):eb==db+1?Jrb.dm(a.k,k,32):(eb!=db||eb!=ib[0])&&Jrb.Ql(a.k,k,db);break;}}u||Jrb.Ql(a.k,k,db);}if(a.i||!Jrb.ep(a.k,k))for(I=0;I<B;I++)Jrb.Jj(a.k,k,Jrb.Hj(a.k,1),1);}}else if(!a.i&&(a.p||a.q==2)){B=Jrb.Kn(a.k,k);B>=1&&Jrb.cm(a.k,k,128,true);B>=2&&Jrb.cm(a.k,k,256,true);B>=3&&Jrb.cm(a.k,k,512,true);B>=4&&Jrb.cm(a.k,k,vtb,true);}}!a.i&&(a.p||a.q==2)&&Jrb.Wo(a.k,true);Jrb.ou(a.k,1);Jrb.Ht(a);Jrb.Jt(a,j,a.p);a.k.t=null;Jrb.xm(a.k,false);if(g){Jrb.Et(a);if(T){for(S=(Q=new Krb.Kcb(T).a.hc().sb(),new Krb.Rcb(Q));S.a.jc();){R=(v=S.a.kc(),v.nc());Jrb.bm(a.k,D[R.a],Jrb._t(R,D),false);}Jrb._o(a.k,0);}}Jrb._o(a.k,0);if(f){K=new Nrb.zy(a.d);E3(a.n,0)&&Nrb.vy(K,a.n);Nrb.gy(K,a.k);g&&Jrb.Bu(a.k);}if(a.p||a.q==2){Jrb.vm(a.k,true);Jrb.gp(a.k);Jrb.hp(a.k);}};Jrb.Lt=function Lt(a,b,c,d,e){Jrb.Kt(a,b,c,0,c.length,d,e);};Jrb.Mt=function Mt(a,b){var c;return b==null?null:Jrb.Nt(a,Hrb.Umb((c=b,hsb.Kab(),c)));};Jrb.Nt=function Nt(a,b){var c,d,e,f,g,h,i,j,k,l,m;g=Zrb.WU(b,0);while(g>0&&b[g-1]==45)g=Zrb.WU(b,g+1);h=g==-1?-1:Zrb.WU(b,g+1);while(h>0&&b[h-1]==45)h=Zrb.WU(b,h+1);if(h==-1)throw h3(new Erb._z("Missing one or both separators ('>')."));if(Zrb.WU(b,h+1)!=-1)throw h3(new Erb._z("Found more than 2 separators ('>')."));m=new Urb.FF();l=0;f=0;c=-1;while(f<b.length){while(f<b.length&&b[f]==46)++f;if(b[f]==40){if(c!=-1)throw h3(new Erb._z('Second open group bracket found before closing first one.'));++f;i=0;for(e=f;e<b.length;e++){if(b[e]==40){++i;}else if(b[e]==41){if(i--==0){c=e;break;}}}}d=f;while(d<b.length&&b[d]!=62&&!(b[d]==46&&(a.o&&c==-1||c==d-1||d+1==b.length||b[d+1]==46)))++d;k=d;if(c==d-1){--k;c=-1;}if(f!=k){j=new Jrb.Gu();Jrb.Kt(a,j,b,f,k,true,true);if(a.q==1&&a.p)return Jrb.Nt(new Jrb.Wt(a.j|2),b);l==0?(Krb.Ei(m.g,j),m.d=-1):l==1?Krb.Ei(m.a,j):(Krb.Ei(m.f,j),m.d=-1);}f=d;while(f<b.length&&b[f]==62){++f;++l;}}return m;};Jrb.Ot=function Ot(a,b){var c,d,e,f;Jrb.Tk(a.k,b)==1&&Jrb.tm(a.k,b,2);for(e=0;e<2;e++){c=Jrb.Ik(a.k,e,b);if(Jrb.sl(a.k,c)){Jrb._l(a.k,c,false);--a.b;}for(f=0;f<Jrb.Gn(a.k,c);f++){d=Jrb.Hn(a.k,c,f);if(a.g[d]){a.g[d]=false;--a.c;}}}};Jrb.Pt=function Pt(a){var b,c,d,e,f,g,h;do{h=false;for(c=0;c<a.k.g;c++){if(a.g[c]){f=false;for(e=0;e<2;e++){b=false;d=Jrb.Ik(a.k,e,c);for(g=0;g<Jrb.Gn(a.k,d);g++){if(c!=Jrb.Hn(a.k,d,g)&&a.g[Jrb.Hn(a.k,d,g)]){b=true;break;}}if(!b){f=true;break;}}if(f){h=true;Jrb.Ot(a,c);}}}}while(h);};Jrb.Qt=function Qt(a){var b,c,d,e,f,g;for(b=0;b<a.k.g;b++){if(a.g[b]){f=true;for(e=0;e<2&&f;e++){c=Jrb.Ik(a.k,e,b);for(g=0;g<Jrb.Gn(a.k,c)&&f;g++){b!=Jrb.Hn(a.k,c,g)&&a.g[Jrb.Hn(a.k,c,g)]&&(f=false);}}if(f){Jrb.tm(a.k,b,8);--a.c;for(d=0;d<2;d++){c=Jrb.Ik(a.k,d,b);if(Jrb.sl(a.k,c)){Jrb._l(a.k,c,false);--a.b;}}}}}};Jrb.Rt=function Rt(a,b){var c,d;if(!Jrb.Vr(Jrb.Dk(a.k,b)))return false;if(Jrb.Dk(a.k,b)==6){if(!Jrb.sl(a.k,b))return false;if(Jrb.mk(a.k,b)>0)return false;}c=Jrb.ok(a.k,b)==null||Jrb.pk(a.k,b)[0]==9?0:Jrb.pk(a.k,b)[0];d=Jrb.Rn(a.k,b)-c;if(d<1)return false;if(Jrb.Dk(a.k,b)==16||Jrb.Dk(a.k,b)==34||Jrb.Dk(a.k,b)==52){if(Jrb.Gn(a.k,b)==2&&Jrb.mk(a.k,b)<=0)return false;return d!=2;}return true;};Jrb.St=function St(a,b){a.f=b;};Jrb.Tt=function Tt(a,b){a.n=b;};Jrb.Ut=function Ut(a,b){if(a.e){!a.r&&(a.r=new Erb.S7());Erb.P7(a.r,' ');Erb.P7(a.r,b);}};Jrb.Vt=function Vt(){Jrb.Wt.call(this,0);};Jrb.Wt=function Wt(a){this.j=a&-4;this.q=a&3;this.o=(a&32)!=0;this.e=(a&64)!=0;this.i=(a&8)!=0;this.d=2;(a&4)!=0&&(this.d|=1);this.i&&(this.d&=-3);};Z3(100,1,{},Jrb.Vt,Jrb.Wt);_.b=0;_.c=0;_.d=0;_.e=false;_.i=false;_.j=0;_.n=0;_.o=false;_.p=false;_.q=0;Drb.HZ=B5(100);Jrb.Xt=function Xt(a,b){return q6(b.b,a.b);};Jrb.Yt=function Yt(a){this.b=a;this.a=1;};Z3(185,1,{185:1,28:1},Jrb.Yt);_.rb=function Zt(a){return Jrb.Xt(this,a);};_.a=0;_.b=0;Drb.EZ=B5(185);Jrb.$t=function $t(a,b,c){if(!a.c){if(a.e.a.length==4){a.c=true;return;}Krb.Ei(a.e,new Jrb.fu(a,b,c));}};Jrb._t=function _t(a,b){var c,d,e;if(a.c)return 3;for(e=new Krb.ueb(a.e);e.a<e.c.a.length;){d=Krb.teb(e);d.a!=wvb&&d.a!=xsb&&(d.a=b[d.a]);}c=false;switch(a.e.a.length){case 2:c=Jrb.cu(a,b);break;case 3:Krb.Ei(a.e,new Jrb.fu(a,xsb,a.b));case 4:c=Jrb.du(a);break;default:return 3;}return a.d^c?1:2;};Jrb.au=function au(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H;t=a.f.k;e=Krb.Ji(a.e,c);H=e.a;l=b[a.a];q=Jrb.Wn(t,H)>0&&t.j[H]==t.e[H];p=Krb.qfb(a.f.a,a.f.t[H]);if(!q&&!p){g=t.i[H][2];if(t.C[g]==1){g=a.f.t[g];(c==0?g>H:a.f.t[t.i[H][1]]>g)&&(d=!d);}return d;}if(q){f=t.i[H][c];g=a.f.t[f];if(g>H){d=!d;t.C[f]==1&&(d=!d);}}if(p){u=t.e[H];n=tX(Drb.NY,Qsb,6,u-1,15,1);o=tX(Drb.NY,Qsb,6,u-1,15,1);v=0;for(r=0,A=0;r<u;r++){g=t.i[H][r];if(g==l)continue;n[A]=g;j=Jrb.Dn(t,H,g);D=Krb.ubb(a.f.s,a.f.t[H]+'_'+a.f.t[g]);o[A++]=D==null?-1:j;D!=null&&++v;}if(q){n[0]<H&&(d=!d);}else {switch(v){case 1:if(Jrb.Wn(t,H)>0){break;}s=o[0]>=0;m=n[s?0:1];w=n[s?1:0];w>H?m>H&&t.C[w]!=1&&(d=!d):m<H&&(d=!d);break;case 2:h=Jrb.bu(a,o[0],H);i=Jrb.bu(a,o[1],H);F=Krb.ubb(a.f.s,H+'_'+a.f.t[h]);G=Krb.ubb(a.f.s,H+'_'+a.f.t[i]);k=a.b;if(c==0){B=F[1]<k?F[1]:F[0];C=G[1]<k?G[1]:G[0];}else {B=F[0]<k?F[1]:F[0];C=G[0]<k?G[1]:G[0];}B<C!=h<i&&(d=!d);}}}return d;};Jrb.bu=function bu(a,b,c){return a.f.k.D[0][b]==c?a.f.k.D[1][b]:a.f.k.D[0][b];};Jrb.cu=function cu(a,b){var c,d;d=false;if(a.f.t==null){a.f.t=tX(Drb.NY,Qsb,6,b.length,15,1);for(c=a.f.t.length;--c>=0;){a.f.t[b[c]]=c;}}d=Jrb.au(a,b,0,d);d=Jrb.au(a,b,1,d);return d;};Jrb.du=function du(a){var b,c,d;c=false;for(b=1;b<a.e.a.length;b++){for(d=0;d<b;d++){Krb.Ji(a.e,d).a>Krb.Ji(a.e,b).a&&(c=!c);Krb.Ji(a.e,d).b>Krb.Ji(a.e,b).b&&(c=!c);}}return c;};Jrb.eu=function eu(a,b,c,d,e,f){this.f=a;if(e!=0&&e!=1){this.c=true;}else {this.a=b;this.b=c;this.d=f;this.e=new Krb.Yi();d!=-1&&Jrb.$t(this,d,c-1);d!=-1&&e==1&&Jrb.$t(this,wvb,c+1);}};Z3(186,1,{186:1},Jrb.eu);_.a=0;_.b=0;_.c=false;_.d=false;Drb.GZ=B5(186);Jrb.fu=function fu(a,b,c){this.c=a;this.a=b;this.b=c;};Z3(149,1,{149:1},Jrb.fu);_.mb=function gu(){return '['+(this.a==wvb?'h':this.a==xsb?'lp':Jrb.sk(this.c.f.k,this.a))+this.b+']';};_.a=0;_.b=0;Drb.FZ=B5(149);Jrb.hu=function hu(a){a.c=new Krb.Yi();};Jrb.iu=function iu(a,b){var c,d,e,f;f=a.c.a.length;if(f==0){Krb.Di(a.c,0,b);return 0;}e=1;while(2*e<=f)e<<=1;d=e;--e;while(d!=0){d>>=1;if(e>=f){e-=d;continue;}c=Erb.e7(b,Krb.Ji(a.c,e));if(c==0)return  -1;if(d==0)break;c<0?e-=d:e+=d;}e<f&&Erb.e7(b,Krb.Ji(a.c,e))>0&&++e;Krb.Di(a.c,e,b);return e;};Jrb.ju=function ju(a,b){var c,d,e,f;f=a.c.a.length;if(f==0)return  -1;e=1;while(2*e<=f)e<<=1;d=e;--e;while(d!=0){d>>=1;if(e>=f){e-=d;continue;}c=Erb.e7(b,Krb.Ji(a.c,e));if(c==0)return e;if(d==0)break;c<0?e-=d:e+=d;}return  -1;};Jrb.ku=function ku(){Jrb.hu(this);};Z3(87,1,{87:1,4:1},Jrb.ku);_.Cb=function lu(a){return Jrb.ju(this,a);};Drb.IZ=B5(87);Jrb.mu=function mu(a){Jrb.Wj(a);a.d=null;};Jrb.nu=function nu(a,b){Jrb.ak(a,b);!!a.d&&(b.T=0);};Jrb.ou=function ou(a,b){var c,d,e,f;Jrb.kn(a,b);if((b&~a.T)==0)return;(a.T&15)!=0&&a.q>1&&Jrb.xh(a.J[0],a.J[1])&&Nrb.gy(new Nrb.zy(0),a);a.c&&(b|=128);for(c=0;c<a.q;c++)a.u[c]&=-67223560;for(d=0;d<a.g;d++)a.F[d]&=-64;e=0;f=0;if((b&32)!=0){e=32;f=1;}else if((b&64)!=0){e=64;f=3;}if((b&128)!=0){e|=128;f|=32;}a.d=new Jrb.fg(a,f);Jrb.ag(a.d);Jrb.bg(a.d);Jrb._f(a.d);Jrb.Fu(a)&&(a.d=new Jrb.fg(a,f));a.T|=24|e;};Jrb.pu=function pu(a,b){return Jrb.Pf(a.d,b);};Jrb.qu=function qu(a,b){return Jrb.Gf(a.d,b);};Jrb.ru=function ru(a){var b;Jrb.ou(a,31);b=a.I&Gsb;switch(a.I&vub){case Fsb:return null;case oub:return b==1?'meso':''+b+' meso diastereomers';case 0:return 'unknown chirality';case 196608:return 'both enantiomers';case bub:return 'this enantiomer';case 327680:return 'this or other enantiomer';case Ftb:return 'two epimers';default:return b==1?'one stereo isomer':''+b+' stereo isomers';}};Jrb.su=function su(a){var b;b=new Jrb.Hu(a.q,a.r);Jrb._j(a,b);return b;};Jrb.tu=function tu(a){var b,c,d,e,f,g,h;Jrb.ou(a,15);g=tX(Drb.NY,Qsb,6,3,15,1);for(c=0;c<a.f;c++){if((a.u[c]&Hub)!=0){h=(a.u[c]&_tb)>>18;h!=0&&(g[h]=$wnd.Math.max(g[h],(a.u[c]&_tb)>>18!=1&&(a.u[c]&_tb)>>18!=2?-1:(a.u[c]&Cub)>>20));}}for(e=0;e<a.g;e++){if(((a.F[e]&3)==1||(a.F[e]&3)==2)&&a.H[e]==1){h=(a.F[e]&768)>>8;h!=0&&(g[h]=$wnd.Math.max(g[h],(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&Dub)>>10));}}f=tX(Drb.NY,lub,7,3,0,2);f[1]=tX(Drb.NY,Qsb,6,1+g[1],15,1);f[2]=tX(Drb.NY,Qsb,6,1+g[2],15,1);for(b=0;b<a.f;b++){if((a.u[b]&Hub)!=0){h=(a.u[b]&_tb)>>18;h!=0&&++f[h][(a.u[b]&_tb)>>18!=1&&(a.u[b]&_tb)>>18!=2?-1:(a.u[b]&Cub)>>20];}}for(d=0;d<a.g;d++){if(((a.F[d]&3)==1||(a.F[d]&3)==2)&&a.H[d]==1){h=(a.F[d]&768)>>8;h!=0&&++f[h][(a.F[d]&768)>>8!=1&&(a.F[d]&768)>>8!=2?-1:(a.F[d]&Dub)>>10];}}return f;};Jrb.uu=function uu(a){var b,c;b=tX(Drb.NY,Qsb,6,a.q,15,1);c=Jrb.Pn(a,b,false,false);return Jrb.vu(a,b,c);};Jrb.vu=function vu(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r;q=tX(Drb.JZ,mvb,27,c,0,1);g=tX(Drb.NY,Qsb,6,c,15,1);j=tX(Drb.NY,Qsb,6,c,15,1);f=tX(Drb.NY,Qsb,6,a.q,15,1);for(e=0;e<a.q;e++)b[e]!=-1&&(f[e]=g[b[e]]++);for(i=0;i<a.r;i++){o=b[a.D[0][i]];p=b[a.D[1][i]];o==p&&o!=-1&&++j[o];}for(r=0;r<c;r++){q[r]=new Jrb.Hu(g[r],j[r]);Jrb.nu(a,q[r]);}for(d=0;d<a.q;d++)b[d]!=-1&&Jrb.Yj(a,q[b[d]],d,0,0);for(h=0;h<a.r;h++){o=b[a.D[0][h]];p=b[a.D[1][h]];o==p&&o!=-1&&Jrb.$j(a,q[o],h,0,0,f,false);}for(l=q,m=0,n=l.length;m<n;++m){k=l[m];Jrb.Ml(k,1);Jrb.Ml(k,2);}return q;};Jrb.wu=function wu(a){Jrb.ou(a,15);!a.d&&(a.f<2||!Jrb.xh(a.J[0],a.J[1]))&&(a.d=new Jrb.eg(a));return !a.d?null:Jrb.Kf(a.d);};Jrb.xu=function xu(a){Jrb.ou(a,15);!a.d&&(a.f<2||!Jrb.xh(a.J[0],a.J[1]))&&(a.d=new Jrb.eg(a));return !a.d?null:Jrb.Hf(a.d);};Jrb.yu=function yu(a){var b,c;Jrb.ou(a,31);c=0;for(b=0;b<a.f;b++)(a.u[b]&3)!=0&&(a.u[b]&4)==0&&++c;return c;};Jrb.zu=function zu(a,b){return Jrb.Of(a.d,b);};Jrb.Au=function Au(a,b){a.c=b;a.T&=-144;};Jrb.Bu=function Bu(a){Jrb.ou(a,31);!!a.d&&Jrb.cg(a.d);};Jrb.Cu=function Cu(a){var b,c;Jrb.ou(a,15);a.L=false;for(b=0;b<a.q;b++){a.u[b]&=Lub;(a.u[b]&3)!=0?a.u[b]|=Gub:a.u[b]&=Kub;}for(c=0;c<a.r;c++)(a.F[c]&3)!=0&&Jrb.Qk(a,c)==2?a.H[c]=386:a.H[c]&=-385;a.T&=-249;};Jrb.Du=function Du(a,b,c){var d,e,f,g,h,i,j,k,l;l=a.u[b]&3;if(l==1||l==2){h=false;if(a.o[b]==2&&a.j[b]==2&&a.n[b][0]==2&&a.n[b][1]==2&&a.C[b]<=7){for(g=0;g<a.j[b];g++){d=a.i[b][g];k=0;j=tX(Drb.NY,Qsb,6,3,15,1);for(i=0;i<a.j[d];i++){j[k]=a.i[d][i];j[k]!=b&&++k;}k==2&&j[0]<j[1]^c[j[0]]<c[j[1]]&&(h=!h);}}else {for(g=1;g<a.j[b];g++){for(i=0;i<g;i++){e=a.i[b][g];f=a.i[b][i];e<f^c[e]<c[f]&&(h=!h);}}}h&&(l=l==1?2:1);}return l;};Jrb.Eu=function Eu(a){var b,c,d,e,f,g;Jrb.fp(a);Jrb.ou(a,31);for(d=0;d<a.f;d++){if(((a.u[d]&_tb)>>18==1||(a.u[d]&_tb)>>18==2)&&((a.u[d]&Hub)==0||(a.u[d]&3)==3))throw h3(new Erb._z(xvb));if((a.u[d]&Fsb)!=0)throw h3(new Erb._z(yvb));if(((a.u[d]&3)==1||(a.u[d]&3)==2)&&a.o[d]==0){b=tX(Drb.LY,Osb,6,a.j[d],15,1);for(f=0;f<a.j[d];f++)b[f]=Jrb.Hk(a,d,a.i[d][f]);for(e=1;e<a.j[d];e++)if(!Jrb.Al(a,a.k[d][e],d))for(g=0;g<e;g++)if(!Jrb.Al(a,a.k[d][g],d)){c=$wnd.Math.abs(Jrb.Wm(b[e],b[g]));if(c<0.08||c>Rub)throw h3(new Erb._z(zvb));}}}};Jrb.Fu=function Fu(a){var b,c,d,e,f,g,h;h=false;for(c=0;c<a.f;c++)((a.u[c]&Hub)==0||(a.u[c]&3)==3)&&(a.u[c]&=Lub);for(f=0;f<a.g;f++)(Jrb.Qk(a,f)!=1||(a.F[f]&3)==0||(a.F[f]&3)==3)&&(a.F[f]&=-32513);if(a.L){if((a.I&vub)!=oub){g=tX(Drb.c3,itb,6,a.f,16,1);for(d=0;d<a.f;d++)(a.u[d]&Hub)!=0&&(a.u[d]&3)!=3&&(a.u[d]&_tb)>>18==1&&(g[d]=true);for(e=0;e<a.f;e++){if((a.u[e]&Hub)!=0&&(a.u[e]&3)!=3){Jrb.Xl(a,e,1,0);h=true;}}for(b=0;b<a.f;b++){if(g[b]){Jrb.bm(a,b,1,false);Jrb.Xl(a,b,1,-1);h=true;}}}a.L=false;}Jrb.Ml(a,1);Jrb.Ml(a,2);return h;};Jrb.Gu=function Gu(){Jrb.Ej();Jrb.jp.call(this);};Jrb.Hu=function Hu(a,b){Jrb.Ej();Jrb.kp.call(this,a,b);};Jrb.Iu=function Iu(a){Jrb.Ej();Jrb.lp.call(this,a);};Z3(27,101,Avb,Jrb.Gu,Jrb.Hu,Jrb.Iu);_.xb=function Ju(a){Jrb.nu(this,a);};_.zb=function Ku(a){Jrb.ou(this,a);};_.c=false;Drb.JZ=B5(27);Jrb.Mu=function Mu(a,b){var c,d,e,f,g,h,i,j,k,l,m;h=new Krb.Yi();Jrb.ou(b,1);j=tX(Drb.c3,itb,6,b.f,16,1);for(c=0;c<b.f;c++){if(a.f[c]&&(Jrb.Dk(a.i,c)==7||Jrb.Dk(a.i,c)==8||Jrb.Dk(a.i,c)==16||Jrb.Dk(a.i,c)==34||Jrb.Dk(a.i,c)==52)){j[c]=true;for(i=0;i<b.j[c];i++){d=b.i[c][i];f=b.k[c][i];l=b.n[c][i];if(b.o[d]!=0&&b.o[c]<l){j[d]=true;Krb.Ei(h,Erb.A6(f));for(k=0;k<b.j[d];k++){e=b.i[d][k];if(!j[e]){j[e]=true;g=b.k[d][k];m=b.n[d][k];if(b.o[d]+2==l+m){Krb.Ei(h,Erb.A6(g));l>=m&&(b.o[e]<m?Jrb.Vu(b,e)&&Jrb.Pu(a,b,e,true,false,j,h):Jrb.Pu(a,b,e,true,true,j,h));m>=l&&Jrb.Vu(b,c)&&Jrb.Pu(a,b,e,false,false,j,h);if(a.f[e]&&(Jrb.Dk(a.i,e)==5||Jrb.Dk(a.i,e)==6||Jrb.Dk(a.i,e)==7||Jrb.Dk(a.i,e)==8||Jrb.Dk(a.i,e)==16||Jrb.Dk(a.i,e)==34||Jrb.Dk(a.i,e)==52)&&b.o[e]<m){l<=2&&m>=2&&Jrb.Vu(b,c)&&Jrb.Nu(a,b,f,g);l>=2&&m<=2&&Jrb.Vu(b,e)&&Jrb.Nu(a,b,g,f);}Krb.Qi(h,h.a.length-1);}j[e]=false;}}Krb.Qi(h,h.a.length-1);j[d]=false;}}j[c]=false;}}};Jrb.Nu=function Nu(a,b,c,d){var e;e=new Jrb._u(a,b);Jrb.$u(e,c,Jrb.Qk(b,c)==1?2:3);Jrb.$u(e,d,Jrb.Qk(b,d)==2?1:2);a.g[c]=true;a.g[d]=true;Krb.ikb(a.e,e)&&(Krb.Ndb(a.d,e),true);};Jrb.Ou=function Ou(a,b){Krb.ikb(a.e,b)&&(Krb.Ndb(a.d,b),true);};Jrb.Pu=function Pu(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p,q,r,s;for(n=0;n<b.j[c];n++){h=b.i[c][n];if(!f[h]){k=b.k[c][n];r=Jrb.Qk(b,k);if(d&&r>=2||!d&&r<=2){f[h]=true;g.add(Erb.A6(k));for(o=0;o<b.j[h];o++){i=b.i[h][o];if(!f[i]){l=b.k[h][o];s=Jrb.Qk(b,l);if(b.o[h]+2==r+s&&(d&&s<=2||!d&&s>=2)){f[i]=true;g.add(Erb.A6(l));if(a.f[i]&&(Jrb.Dk(a.i,i)==5||Jrb.Dk(a.i,i)==6||Jrb.Dk(a.i,i)==7||Jrb.Dk(a.i,i)==8||Jrb.Dk(a.i,i)==16||Jrb.Dk(a.i,i)==34||Jrb.Dk(a.i,i)==52)&&(!d||Jrb.Vu(b,i))){m=new Jrb._u(a,b);for(p=0;p<g.a.length;p++){j=(Hrb.fnb(p,g.a.length),g.a[p]).a;q=p<2?d^(p&1)==0:e^(p&1)==0;q?Jrb.$u(m,j,Jrb.Qk(b,j)==1?2:3):Jrb.$u(m,j,Jrb.Qk(b,j)==2?1:2);a.g[j]=true;}Krb.ikb(a.e,m)&&(Krb.Ndb(a.d,m),true);}else {Jrb.Pu(a,b,i,d,e,f,g);}Krb.Qi(g,g.a.length-1);f[i]=false;}}}Krb.Qi(g,g.a.length-1);f[h]=false;}}}return false;};Jrb.Qu=function Qu(a){var b,c,d,e,f,g,h,i,j,k,l;a.b=tX(Drb.NY,Qsb,6,a.i.f,15,1);h=tX(Drb.NY,Qsb,6,a.i.f,15,1);d=tX(Drb.c3,itb,6,a.i.g,16,1);l=0;for(c=0;c<a.i.g;c++){if(!d[c]&&a.g[c]){++l;a.b[Jrb.Ik(a.i,0,c)]=l;a.b[Jrb.Ik(a.i,1,c)]=l;d[c]=true;for(j=0;j<2;j++){b=Jrb.Ik(a.i,j,c);a.b[b]=l;g=0;i=0;h[0]=b;while(g<=i){for(k=0;k<Jrb.Gn(a.i,h[g]);k++){f=Jrb.Hn(a.i,h[g],k);if(!d[f]&&a.g[f]){d[f]=true;e=Jrb.Fn(a.i,h[g],k);if(a.b[e]==0){a.b[e]=l;h[++i]=e;}}}++g;}}}}a.j=l;};Jrb.Ru=function Ru(a){var b,c,d;a.n=tX(Drb.NY,Qsb,6,a.j,15,1);a.k=tX(Drb.NY,Qsb,6,a.j,15,1);a.o=tX(Drb.NY,Qsb,6,a.j,15,1);for(b=0;b<a.i.f;b++){if(a.b[b]!=0){d=a.b[b]-1;a.a!=null&&(a.k[d]+=a.a[b]);a.c!=null&&(a.o[d]+=a.c[b]);}}for(c=0;c<a.i.g;c++){a.g[c]&&Jrb.Qk(a.i,c)==2&&(a.n[a.b[Jrb.Ik(a.i,0,c)]-1]+=2);}};Jrb.Su=function Su(a){var b,c,d,e;for(d=0;d<a.i.r;d++){for(e=0;e<2;e++){b=Jrb.Ik(a.i,e,d);c=Jrb.Ik(a.i,1-e,d);if(Jrb.Dk(a.i,b)==1&&Jrb.wk(a.i,b)>1&&Jrb.Dk(a.i,c)>1&&a.b[c]!=0){if(Jrb.wk(a.i,b)==2){a.a==null&&(a.a=tX(Drb.NY,Qsb,6,a.i.q,15,1));++a.a[c];}else {a.c==null&&(a.c=tX(Drb.NY,Qsb,6,a.i.q,15,1));++a.c[c];}Jrb.Cl(a.i,b);}}}(a.a!=null||a.c!=null)&&Jrb.gk(a.i);};Jrb.Tu=function Tu(a){var b;a.e=new Krb.nkb();a.d=new Krb.Zdb();Jrb.Ou(a,new Jrb._u(a,a.i));b=Jrb.su(a.i);while(!Krb.Rdb(a.d)){Jrb.Zu(Krb.Tdb(a.d),b);Jrb.Mu(a,b);if(a.e.a.c>=Jrb.Lu){Erb.X7();'Tautomer count exceeds maximum: '+Jrb.Kf(new Jrb.eg(a.i));break;}}};Jrb.Uu=function Uu(a){var b,c,d,e,f,g,h,i,j,k;if(a.e.a.c==1)return a.i;j=Jrb.su(a.i);Jrb.vm(j,true);Jrb.ou(j,7);for(e=0;e<j.g;e++){if(a.g[e]){j.H[e]=1;j.T=0;Jrb.sm(j,e,3,true);}}for(c=0;c<j.f;c++){if(a.b[c]!=0&&Jrb._n(a.i,c)<4){Jrb.gn(j,c);j.u[c]&=Kub;j.T&=7;Jrb.Xl(j,c,0,-1);}}h=tX(Drb.NY,Qsb,6,a.j,15,1);i=tX(Drb.NY,Qsb,6,a.j,15,1);d=new Jrb.eg(j).e;for(b=0;b<j.f;b++){if(a.b[b]!=0){k=a.b[b]-1;if(i[k]<d[b]){i[k]=d[b];h[k]=b;}}}for(f=0;f<a.j;f++){g=''+a.n[f]+'|'+a.k[f]+'|'+a.o[f];Jrb.Vl(j,h[f],g);}return j;};Jrb.Vu=function Vu(a,b){if(a.e[b]-Jrb._n(a,b)+Jrb.Wn(a,b)<=0)return false;if(Jrb.ql(a,b))return true;if(a.o[b]!=0)return false;return true;};Jrb.Wu=function Wu(a){var b,c,d,e,f,g,h,i;Jrb.ou(a.i,1);h=a.i.f;do--h;while(h>=0&&Jrb.Dk(a.i,h)==1);for(b=0;b<h;b++){if(Jrb.Dk(a.i,b)==1){Jrb.Gm(a.i,b,h);do--h;while(Jrb.Dk(a.i,h)==1);}}if(h==a.i.f-1)return;g=tX(Drb.c3,itb,6,a.i.g,16,1);for(f=0;f<a.i.g;f++){c=Jrb.Ik(a.i,0,f);d=Jrb.Ik(a.i,1,f);(Jrb.Dk(a.i,c)==1||Jrb.Dk(a.i,d)==1)&&(g[f]=true);}i=a.i.g;do--i;while(i>=0&&g[i]);for(e=0;e<i;e++){if(g[e]){Jrb.Hm(a.i,e,i);g[e]=false;do--i;while(g[i]);}}};Jrb.Xu=function Xu(a){var b,c;this.i=Jrb.su(a);Jrb.Wu(this);Jrb.ou(this.i,7);this.g=tX(Drb.c3,itb,6,this.i.g,16,1);this.f=tX(Drb.c3,itb,6,this.i.f,16,1);for(b=0;b<this.i.f;b++){c=Jrb.Um(Jrb.Dk(this.i,b))[0];this.f[b]=Jrb._n(this.i,b)<c;}Jrb.Tu(this);Jrb.Qu(this);Jrb.Su(this);Jrb.Ru(this);};Z3(205,1,{},Jrb.Xu);_.j=0;Jrb.Lu=Bvb;Drb.LZ=B5(205);Jrb.Yu=function Yu(a,b){return Zrb.gV(a.a,b.a);};Jrb.Zu=function Zu(a,b){var c,d;for(d=0;d<a.b.i.g;d++){if(a.b.g[d]){c=3&a.a[d>>4]>>2*(d&15);Jrb.tm(b,d,c==1?1:c==2?a.b.g[d]&&!Jrb.Po(a.b.i,d)?386:2:c==3?4:16);}}};Jrb.$u=function $u(a,b,c){var d,e;d=b>>4;e=2*(b&15);a.a[d]&=~(3<<e);a.a[d]|=c<<e;};Jrb._u=function _u(a,b){var c;this.b=a;this.a=tX(Drb.NY,Qsb,6,(a.i.g+15)/16|0,15,1);for(c=0;c<a.i.g;c++)this.a[c>>4]|=$wnd.Math.min(3,Jrb.Qk(b,c))<<2*(c&15);};Z3(137,1,{137:1,28:1},Jrb._u);_.rb=function av(a){return Jrb.Yu(this,a);};Drb.KZ=B5(137);Jrb.bv=function bv(a){var b,c,d,e,f;c=Erb.o7(a,A7(38));if(c==-1)return a;e=0;f=new Erb.I7();while(c!=-1){Erb.H7(f,(Hrb.lnb(e,c,Hrb.rnb(a).length),Hrb.rnb(a).substr(e,c-e)));Hrb.mnb(c+1,Hrb.rnb(a).length);if(Hrb.rnb(a).charCodeAt(c+1)==38){f.a+='&';e=c+2;}else {b=(Hrb.mnb(c+1,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(c+1));b=b-(b<65?48:b<97?65:97);d=(Hrb.mnb(c+2,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(c+2));d=d-(d<65?48:d<97?65:97);Erb.F7(f,16*b+d&Gsb);e=c+3;}c=Erb.p7(a,A7(38),e);}Erb.H7(f,(Hrb.mnb(e,Hrb.rnb(a).length+1),Hrb.rnb(a).substr(e)));return f.a;};Jrb.cv=function cv(a){var b,c;c=new Erb.I7();for(b=0;b<Hrb.rnb(a).length;b++){switch(Hrb.mnb(b,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(b)){case 38:c.a+='&&';break;case 9:c.a+='&09';break;case 10:c.a+='&0A';break;case 32:c.a+='&20';break;default:Erb.F7(c,(Hrb.mnb(b,Hrb.rnb(a).length),Hrb.rnb(a).charCodeAt(b)));}}return c.a;};Jrb.dv=function dv(a,b,c){a.e[0].a=b;a.e[0].b=c;};Jrb.ev=function ev(b,c){try{b.a=g6(c);}catch(a){a=g3(a);if(!zY(a,39))throw h3(a);}};Jrb.fv=function fv(b,c){try{b.b=Erb.V5(c,_ub,xsb);}catch(a){a=g3(a);if(!zY(a,39))throw h3(a);}};Jrb.gv=function gv(a,b,c,d){a.c=b;a.a=c;a.b=d;};Jrb.hv=function hv(b,c){try{b.e[0].a=g6(c);}catch(a){a=g3(a);if(!zY(a,39))throw h3(a);}};Jrb.iv=function iv(b,c){try{b.e[0].b=g6(c);}catch(a){a=g3(a);if(!zY(a,39))throw h3(a);}};Jrb.jv=function jv(){Jrb.lv.call(this,new Wrb.BG());};Jrb.kv=function kv(a){var b,c,d,e;Jrb.jv.call(this);b=0;while(b!=-1){c=Hrb.rnb(a).indexOf('="',b);if(c==-1)break;d=(Hrb.lnb(b+1,c,Hrb.rnb(a).length),Hrb.rnb(a).substr(b+1,c-(b+1)));b=Hrb.rnb(a).indexOf('"',c+2);e=b==-1?(Hrb.mnb(c+1,Hrb.rnb(a).length+1),Hrb.rnb(a).substr(c+1)):(Hrb.lnb(c+1,b,Hrb.rnb(a).length),Hrb.rnb(a).substr(c+1,b-(c+1)));Erb.k7(d,'text')?this.c=Jrb.bv(e):Erb.k7(d,'x')?Jrb.hv(this,e):Erb.k7(d,'y')?Jrb.iv(this,e):Erb.k7(d,'size')?Jrb.ev(this,e):Erb.k7(d,'style')&&Jrb.fv(this,e);}};Jrb.lv=function lv(a){this.c='';this.a=9;this.b=0;this.e=tX(Drb.e_,Bsb,52,1,0,1);this.e[0]=a;};Z3(139,169,{169:1},Jrb.jv,Jrb.kv);_.nb=function mv(){var a;a=new Jrb.jv();Jrb.gv(a,this.c,this.a,this.b);Jrb.dv(a,this.e[0].a,this.e[0].b);a.d=this.d;return a;};_.ob=function nv(){var a;a=new Erb.S7();Erb.P7(a,' text="'+Jrb.cv(this.c)+'"');Erb.P7(a,' x="'+this.e[0].a+'"');Erb.P7(a,' y="'+this.e[0].b+'"');this.a!=9&&Erb.P7(a,' size="'+gsb.s8(new gsb.v8(this.a,new gsb.aab()))+'"');this.b!=0&&Erb.P7(a,' style="'+this.b+'"');return a.a;};_.pb=function ov(){return 'text';};_.a=0;_.b=0;Drb.MZ=B5(139);Jrb.pv=function pv(a){a.b=new Krb.Yi();a.a=new Krb.Yi();};Jrb.qv=function qv(a,b){var c,d;c=Jrb.iu(a,b);if(c==-1)return  -1;d=a.b.a.length;Krb.Ei(a.b,b);Krb.Di(a.a,c,new Erb.o6(d));return d;};Jrb.rv=function rv(){Jrb.ku.call(this);Jrb.pv(this);};Z3(247,87,{87:1,4:1},Jrb.rv);_.Cb=function sv(a){var b;b=Jrb.ju(this,a);if(b==-1)return  -1;return Krb.Ji(this.a,b).a;};Drb.NZ=B5(247);Lrb.tv=function tv(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N;K=Jrb.jo(a.e);L=Krb.Ji(K.j,b).length;I=Krb.Ji(K.i,b);J=Krb.Ji(K.j,b);C=true;for(r=1;r<L;r++){if(Lrb.Iv(a.b,J[r])!=Lrb.Iv(a.b,J[0])){C=false;break;}}if(C){f=(L-2)*Cvb/L;for(s=0;s<L;s++)Lrb.yv(a,I[s],J[s],J[s==0?L-1:s-1],f);return;}G=tX(Drb.MY,Pub,6,L,15,1);e=0;for(t=0;t<L;t++){G[t]=Jrb.wn(a.e,I[t])==0?1.9106119386631804:Jrb.wn(a.e,I[t])==1?2.094499897810209:Cvb;e+=G[t];}d=((L-2)*Cvb-e)/L;for(u=0;u<L;u++)G[u]+=d;k=tX(Drb.MY,Pub,6,L,15,1);for(v=1;v<L;v++)k[v]=k[v-1]+Cvb-G[v];h=tX(Drb.MY,Pub,6,L,15,1);for(g=0;g<100;g++){M=0;N=0;for(s=0;s<L;s++){M+=Lrb.Iv(a.b,J[s])*$wnd.Math.sin(k[s]);N+=Lrb.Iv(a.b,J[s])*$wnd.Math.cos(k[s]);}n=Jrb.Vm(0,0,M,N);p=$wnd.Math.sqrt(M*M+N*N);F=-1;D=0;for(w=0;w<L;w++){A=w==0?L-1:w-1;B=w+1==L?0:w+1;i=Jrb.Wm(k[w],k[A]);j=Jrb.Wm(k[B],k[w]);H=Jrb.Wm(G[w],G[B]);h[w]=2*i-2*j+2*H;o=$wnd.Math.cos(k[w]-Dvb-n);m=p*o-Evb*h[w];if($wnd.Math.abs(m)>$wnd.Math.abs(D)){D=m;F=w;}}l=$wnd.Math.exp(-5*g/100);k[F]+=l*D;}for(q=0;q<L;q++){A=q==0?L-1:q-1;c=k[A]+Cvb-k[q];c>mtb&&(c-=mtb);Lrb.yv(a,I[q],J[A],J[q],c);}};Lrb.uv=function uv(a,b){var c,d,e,f,g,h,i,j,k;i=Jrb.jo(a.e);j=Krb.Ji(i.j,b).length;h=Krb.Ji(i.j,b);k=false;for(e=0;e<j;e++){if(a.f[h[e]]==0){k=true;break;}}if(k){c=(180*j-360)/j*0.01745329238474369;g=Krb.Ji(i.i,b);for(f=0;f<j;f++){Lrb.yv(a,g[f],h[f==0?j-1:f-1],h[f],c);}if(j<=4){for(d=0;d<j;d++)a.f[h[d]]+=j;}}};Lrb.vv=function vv(a,b,c,d){var e,f,g,h;f=-1;g=-1;for(h=0;h<Jrb.rn(a.e,b);h++){e=Jrb.Fn(a.e,b,h);if(e==c||e==d){if(g==-1){g=h;}else {f=h;break;}}}return a.a[b][f][g];};Lrb.wv=function wv(a,b,c,d){return c<d?a.a[b][d][c]:a.a[b][c][d];};Lrb.xv=function xv(a,b){var c,d,e,f,g,h,i;d=tX(Drb.c3,itb,6,Jrb.Gn(a.e,b),16,1);i=0;for(e=0;e<Jrb.Gn(a.e,b);e++){h=0;g=-1;for(f=0;f<Jrb.Gn(a.e,b);f++){if(!d[f]){c=Jrb.Hn(a.e,b,f);if(h<a.f[c]){h=a.f[c];g=f;}}}if(h==0)return i;i<<=8;i+=h;d[g]=true;}return i;};Lrb.yv=function yv(a,b,c,d,e){var f,g,h,i;f=-1;g=-1;for(i=0;i<Jrb.rn(a.e,b);i++){h=Jrb.Hn(a.e,b,i);if(h==c||h==d){if(f==-1){f=i;}else {g=i;break;}}}if(a.a[b][g][f]==0){a.a[b][g][f]=e;a.d[b]+=e;++a.c[b];}};Lrb.zv=function zv(a,b,c,d,e){var f,g,h,i,j,k;f=Jrb.rn(a.e,b);i=Jrb.Gn(a.e,b);for(j=1;j<f;j++){g=j<i?a.f[Jrb.Hn(a.e,b,j)]:0;for(k=0;k<j;k++){if(a.a[b][j][k]==0){h=k<i?a.f[Jrb.Hn(a.e,b,k)]:0;(g==c&&h==d||g==d&&h==c)&&(a.a[b][j][k]=e);}}}};Lrb.Av=function Av(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;this.b=b;this.e=a;this.a=tX(Drb.MY,ssb,389,this.e.f,0,3);for(e=0;e<this.e.f;e++){this.a[e]=tX(Drb.MY,{389:1,4:1,10:1,5:1,8:1},40,Jrb.rn(this.e,e),0,2);for(i=1;i<Jrb.rn(this.e,e);i++)this.a[e][i]=tX(Drb.MY,Pub,6,i,15,1);}this.c=tX(Drb.NY,Qsb,6,this.e.f,15,1);this.d=tX(Drb.MY,Pub,6,this.e.f,15,1);this.f=tX(Drb.NY,Qsb,6,this.e.g,15,1);w=Jrb.jo(this.e);m=tX(Drb.c3,itb,6,w.i.a.length,16,1);Jrb.rr(w,m,tX(Drb.c3,itb,6,w.i.a.length,16,1),tX(Drb.NY,Qsb,6,w.i.a.length,15,1),true);for(h=3;h<=7;h++){for(r=0;r<w.i.a.length;r++){A=Krb.Ji(w.j,r).length;A==h&&(m[r]?Lrb.tv(this,r):A<=4&&Lrb.uv(this,r));}}l=tX(Drb.c3,itb,6,this.e.f,16,1);for(q=0;q<w.i.a.length;q++)if(m[q])for(t=Krb.Ji(w.i,q),u=0,v=t.length;u<v;++u){s=t[u];l[s]=true;}f=BX(nX(Drb.NY,1),Qsb,6,15,[0,0,1,3,6,10,15,21]);for(d=0;d<this.e.f;d++){g=Jrb.rn(this.e,d);if(g>4){for(i=1;i<g;i++)for(o=0;o<i;o++)this.a[d][i][o]=Dvb;this.c[d]=f[g];continue;}if(this.c[d]==f[g])continue;if(Jrb.Oo(this.e,d)&&(l[d]||Jrb.An(this.e,d)<=4&&Jrb.wn(this.e,d)>0)){if(g>2){if(this.c[d]==1){Jrb.Dk(this.e,d)<=14?c=(Fvb-this.d[d])/2:c=Gvb+(Gvb-this.d[d])*0.18000000715255737;}else {c=Fvb-this.d[d];if(g>3){if(this.c[d]==2){n=tX(Drb.c3,itb,6,Jrb.rn(this.e,d),16,1);for(j=1;j<g;j++){for(p=0;p<j;p++){if(this.a[d][j][p]!=0){n[j]=!n[j];n[p]=!n[p];}}}for(k=0;k<g;k++){if(n[k]){for(p=k+1;p<g;p++){if(n[p]){this.a[d][p][k]=c;break;}}break;}}}c=Dvb;}}for(i=1;i<g;i++)for(o=0;o<i;o++)this.a[d][i][o]==0&&(this.a[d][i][o]=c);}}else if(Jrb.Oo(this.e,d)&&Jrb.An(this.e,d)<=4){switch(Lrb.xv(this,d)){case 771:Lrb.zv(this,d,0,3,2.0653998851776123);Lrb.zv(this,d,0,0,1.9814722631346626);break;case 1028:Lrb.zv(this,d,0,4,1.9797999858856201);Lrb.zv(this,d,0,0,1.94691481878138);break;case 393987:Lrb.zv(this,d,0,3,Hvb);Lrb.zv(this,d,0,6,Hvb);Lrb.zv(this,d,3,3,1.7229016938441077);break;case 459779:Lrb.zv(this,d,0,3,Ivb);Lrb.zv(this,d,0,4,Ivb);Lrb.zv(this,d,0,7,Ivb);Lrb.zv(this,d,3,4,1.9322539839360076);break;case 525316:Lrb.zv(this,d,0,4,Jvb);Lrb.zv(this,d,0,8,Jvb);Lrb.zv(this,d,4,4,1.99944913298566);case 394758:Lrb.zv(this,d,0,6,2.526099920272827);break;case 460550:Lrb.zv(this,d,0,6,Kvb);Lrb.zv(this,d,0,7,Kvb);break;case 526087:Lrb.zv(this,d,0,7,Lvb);Lrb.zv(this,d,0,8,Lvb);break;case 526344:Lrb.zv(this,d,0,8,2.186300039291382);break;case 50529027:Lrb.zv(this,d,3,3,2.4189000129699707);break;case 67371779:Lrb.zv(this,d,3,4,2.2298998832702637);break;case 67372036:Lrb.zv(this,d,4,4,2.094399929046631);break;case 101057283:Lrb.zv(this,d,3,6,1.839926051241747);Lrb.zv(this,d,3,3,2.9061476191098734);break;case 117834755:Lrb.zv(this,d,3,4,2.812249087174905);Lrb.zv(this,d,3,7,1.7910569124592968);Lrb.zv(this,d,4,6,2.1224948975613245);break;case 134677507:Lrb.zv(this,d,3,4,2.642428360523752);Lrb.zv(this,d,3,8,2.027723514585844);Lrb.zv(this,d,4,7,2.251474717631936);break;case 117900035:Lrb.zv(this,d,3,7,2.109753935530918);Lrb.zv(this,d,3,3,3.1052897491356646);break;case 117900292:Lrb.zv(this,d,4,7,2.090729910747413);Lrb.zv(this,d,4,4,2.551671293386306);break;case 134743044:Lrb.zv(this,d,4,8,2.139250042271712);Lrb.zv(this,d,4,4,2.3520055858942612);}}else {c=Jrb.Dk(this.e,d)>10?Mvb:Jrb.wn(this.e,d)==2?Cvb:Jrb.Fo(this.e,d,true)?Nvb:Jrb.wn(this.e,d)==0?Mvb:Nvb;for(i=1;i<g;i++)for(o=0;o<i;o++)this.a[d][i][o]=c;}}};Z3(323,1,{},Lrb.Av);Drb.OZ=B5(323);Lrb.Hv=function Hv(){Lrb.Hv=_3;Lrb.Fv=BX(nX(Drb.c3,1),itb,6,16,[false,false,false,false,false,true,true,true,true,false,false,false,false,false,false,true,true]);};Lrb.Iv=function Iv(a,b){return a.a[b];};Lrb.Jv=function Jv(a){Lrb.Hv();var b,c;Jrb.ou(a,7);this.a=tX(Drb.MY,Pub,6,a.r,15,1);this.b=tX(Drb.MY,Pub,6,a.r,15,1);for(b=0;b<a.r;b++){c=Lrb.Mv(a,b);if(c==-1){this.a[b]=Lrb.Nv(a,b);this.b[b]=Lrb.Ov(a,b);}else {this.a[b]=c==-1?2.000499963760376:Lrb.Dv[c];this.b[b]=c==-1?1:Lrb.Ev[c];}}};Lrb.Kv=function Kv(a,b){if(b>=a.f)return 0;if(b<a.f&&Jrb.Fr(a.p,b)&&a.C[b]==6&&a.s[b]!=0)return 1;return a.o[b];};Lrb.Lv=function Lv(a,b,c,d,e,f,g){var h,i,j,k,l;k=d<Lrb.Fv.length&&Lrb.Fv[d]?f<<8:0;l=e<Lrb.Fv.length&&Lrb.Fv[e]?g<<8:0;h=k+d;i=l+e;j=c?0:b?4+a:a;return (j<<24)+(h<i?(h<<12)+i:(i<<12)+h);};Lrb.Mv=function Mv(a,b){var c,d,e,f;c=a.D[0][b];d=a.D[1][b];e=a.C[c];f=a.C[d];return Lrb.Qv(Lrb.Lv(Jrb.Qk(a,b),b<a.g&&Jrb.Gr(a.p,b),b<a.g&&(Jrb.Kr(a.p,b)||a.H[b]==8),e,f,Lrb.Kv(a,c),Lrb.Kv(a,d)));};Lrb.Nv=function Nv(a,b){var c,d;c=Jrb.Dk(a,a.D[0][b]);d=Jrb.Dk(a,a.D[1][b]);return (c<(Lrb.nx(),Lrb.lx).length?Lrb.lx[c]:c<Lrb.mx.length?Ovb*Lrb.mx[c]:Pvb)+(d<Lrb.lx.length?Lrb.lx[d]:d<Lrb.mx.length?Ovb*Lrb.mx[d]:Pvb);};Lrb.Ov=function Ov(a,b){var c,d;c=Jrb.Dk(a,a.D[0][b]);d=Jrb.Dk(a,a.D[1][b]);return (c<(Lrb.nx(),Lrb.lx).length?Qvb:0.125)+(d<Lrb.lx.length?Qvb:0.125);};Lrb.Pv=function Pv(){var b,c,d,e,f,g,h;if(!Lrb.Gv){b=(h=Erb.v7('2312\n4099\t2.403\t0.282\t466\n4100\t1.407\t0.2\t28\n4103\t1.561\t0.023\t5\n4104\t1.42\t0.092\t491\n4105\t1.313\t0.149\t32\n4107\t2.677\t0.401\t190\n4108\t2.118\t0.244\t325\n4109\t1.978\t0.26\t968\n4113\t1.819\t0.033\t4\n4115\t3.194\t0.482\t381\n4116\t2.771\t0.32\t198\n4117\t2.398\t0.235\t145\n4118\t2.317\t0.22\t208\n4119\t2.034\t0.212\t34\n4120\t2.033\t0.286\t103\n4121\t1.959\t0.264\t122\n4122\t1.918\t0.267\t664\n4123\t1.893\t0.257\t185\n4124\t1.877\t0.218\t259\n4125\t2.05\t0.257\t556\n4126\t2.012\t0.239\t129\n4127\t1.979\t0.292\t196\n4133\t3.305\t0.429\t34\n4134\t2.593\t0.367\t83\n4135\t2.74\t0.174\t706\n4136\t2.462\t0.196\t611\n4137\t2.207\t0.264\t78\n4138\t2.16\t0.225\t359\n4140\t2.004\t0.188\t2204\n4141\t1.916\t0.256\t610\n4142\t1.95\t0.234\t126\n4143\t2.3\t0.213\t81\n4144\t2.187\t0.116\t6\n4145\t2.103\t0.153\t21\n4146\t2.031\t0.308\t83\n4147\t1.752\t0.443\t21\n4151\t3.49\t0.4\t83\n4152\t3.017\t0.331\t40\n4153\t2.932\t0.276\t176\n4154\t2.39\t0.34\t24\n4155\t3.124\t0.377\t16\n4156\t2.907\t0.232\t198\n4158\t2.857\t0.237\t142\n4159\t2.687\t0.244\t49\n4160\t2.787\t0.359\t29\n4161\t2.785\t0.306\t37\n4162\t2.623\t0.298\t175\n4163\t2.812\t0.154\t70\n4164\t2.745\t0.281\t100\n4165\t2.719\t0.184\t37\n4166\t2.683\t0.361\t231\n4167\t2.655\t0.253\t198\n4168\t2.546\t0.208\t37\n4169\t2.259\t0.231\t190\n4170\t2.043\t0.289\t251\n4171\t2.001\t0.221\t391\n4172\t1.984\t0.164\t885\n4173\t1.904\t0.228\t1074\n4174\t2.036\t0.224\t250\n4175\t1.955\t0.184\t6\n4176\t2.35\t0.15\t12\n4177\t2.367\t0.216\t7\n4178\t2.47\t0.186\t11\n4179\t0.95\t-1.0\t1\n4186\t2.88\t0.309\t39\n4188\t2.804\t0.289\t136\n4189\t2.172\t0.082\t12\n4190\t2.321\t0.0\t2\n4359\t1.503\t0.106\t24\n4360\t1.425\t0.052\t40\n12291\t2.861\t0.27\t1069\n12292\t2.773\t0.086\t4\n12293\t2.507\t0.251\t27\n12294\t2.769\t0.373\t370\n12299\t3.1\t0.221\t116\n12300\t2.844\t0.146\t9\n12301\t3.636\t0.002\t3\n12302\t2.779\t0.318\t189\n12307\t3.639\t0.518\t25\n12308\t3.205\t0.245\t16\n12317\t3.036\t0.064\t4\n12318\t2.88\t-1.0\t1\n12319\t2.615\t0.864\t3\n12320\t2.857\t0.129\t247\n12325\t3.557\t0.41\t18\n12326\t3.273\t0.208\t23\n12330\t3.496\t0.025\t2\n12333\t3.012\t0.276\t2\n12334\t2.679\t0.0\t2\n12338\t2.886\t0.196\t80\n12339\t2.871\t0.221\t18\n12343\t3.628\t0.266\t9\n12344\t3.37\t0.145\t35\n12346\t2.859\t0.0\t4\n12352\t2.722\t0.017\t4\n12354\t3.317\t0.016\t4\n12355\t2.714\t0.006\t2\n12357\t2.703\t0.005\t2\n12365\t2.615\t-1.0\t1\n12367\t2.729\t0.0\t2\n12368\t2.841\t0.0\t2\n12369\t2.889\t0.082\t5\n12370\t2.913\t0.126\t64\n12549\t3.003\t0.175\t18\n12550\t2.797\t0.21\t1156\n12805\t2.63\t0.0\t2\n12806\t2.939\t0.243\t85\n16388\t2.29\t0.104\t57\n16389\t2.022\t0.097\t6\n16390\t2.097\t0.127\t21\n16391\t1.771\t0.395\t281\n16392\t1.648\t0.242\t634\n16393\t1.567\t0.555\t264\n16395\t2.926\t0.104\t14\n16397\t2.368\t-1.0\t1\n16398\t3.269\t-1.0\t1\n16399\t2.211\t0.15\t6\n16400\t2.1\t-1.0\t1\n16401\t2.065\t0.622\t37\n16403\t3.355\t0.087\t11\n16404\t2.9\t0.136\t2\n16416\t2.399\t0.154\t3\n16417\t2.06\t0.0\t2\n16418\t2.195\t-1.0\t1\n16419\t2.167\t0.95\t16\n16421\t3.378\t-1.0\t1\n16422\t2.886\t0.031\t2\n16429\t2.375\t0.124\t6\n16435\t2.635\t-1.0\t1\n16436\t2.399\t-1.0\t1\n16437\t2.417\t0.96\t23\n16439\t3.716\t0.078\t2\n16442\t3.052\t0.0\t4\n16463\t2.64\t-1.0\t1\n16466\t2.23\t-1.0\t1\n16474\t3.061\t0.0\t4\n16476\t3.017\t0.004\t8\n16645\t1.954\t-1.0\t1\n16646\t2.276\t0.281\t56\n16647\t1.966\t0.14\t51\n16648\t1.843\t0.124\t41\n16902\t1.739\t0.003\t2\n16903\t2.003\t-1.0\t1\n20491\t2.681\t-1.0\t1\n20492\t2.402\t0.287\t12\n20493\t2.373\t0.17\t6\n20499\t3.054\t0.395\t11\n20500\t3.076\t0.081\t2\n20501\t2.3\t1.168\t20\n20502\t2.221\t0.057\t3\n20503\t1.964\t-1.0\t1\n20504\t2.515\t0.236\t21\n20505\t1.97\t0.231\t70\n20506\t2.056\t0.352\t138\n20507\t2.553\t0.302\t81\n20508\t2.489\t0.321\t24\n20509\t2.215\t0.363\t36\n20510\t2.786\t0.026\t6\n20511\t2.418\t0.106\t10\n20517\t3.193\t0.063\t3\n20519\t2.598\t0.187\t9\n20520\t2.376\t0.309\t9\n20522\t2.708\t0.265\t154\n20524\t2.526\t0.319\t123\n20525\t2.262\t0.289\t122\n20526\t2.502\t0.287\t18\n20527\t2.365\t0.284\t11\n20528\t2.966\t0.038\t6\n20529\t2.584\t0.061\t7\n20530\t2.771\t0.387\t5\n20531\t1.29\t0.001\t12\n20535\t3.368\t0.154\t5\n20536\t3.023\t0.074\t6\n20537\t2.913\t0.0\t2\n20538\t3.021\t0.331\t2\n20540\t2.651\t0.201\t5\n20543\t2.735\t0.344\t5\n20546\t2.531\t0.277\t7\n20549\t3.267\t-1.0\t1\n20550\t2.329\t0.34\t7\n20552\t2.499\t0.02\t6\n20553\t2.703\t0.218\t20\n20554\t2.693\t0.292\t46\n20555\t2.639\t0.27\t40\n20556\t2.066\t0.059\t39\n20557\t2.599\t0.324\t183\n20558\t2.048\t0.149\t134\n20559\t2.327\t0.386\t34\n20560\t2.179\t-1.0\t1\n20561\t2.619\t-1.0\t1\n20562\t2.967\t-1.0\t1\n20570\t2.665\t0.007\t2\n20572\t2.63\t0.19\t9\n20742\t1.87\t0.155\t40\n24587\t3.525\t0.393\t28\n24588\t2.763\t0.313\t380\n24589\t2.462\t0.308\t5373\n24595\t4.077\t0.356\t19\n24596\t3.17\t0.362\t45\n24597\t2.829\t0.525\t432\n24598\t2.656\t0.164\t627\n24599\t2.654\t0.158\t57\n24600\t2.57\t0.26\t309\n24601\t2.625\t0.288\t208\n24602\t2.45\t0.223\t1004\n24603\t2.182\t0.257\t1071\n24604\t2.355\t0.185\t1159\n24605\t2.384\t0.162\t130\n24606\t2.479\t0.188\t1375\n24607\t2.547\t0.23\t805\n24613\t3.789\t0.188\t2\n24614\t3.484\t0.291\t8\n24615\t3.089\t0.34\t499\n24616\t2.809\t0.177\t912\n24617\t2.665\t0.147\t106\n24618\t2.632\t0.3\t598\n24620\t2.419\t0.207\t1127\n24621\t2.466\t0.217\t582\n24622\t2.456\t0.168\t1672\n24623\t2.573\t0.135\t12\n24624\t2.753\t0.503\t34\n24625\t2.695\t0.487\t478\n24626\t2.644\t0.444\t4321\n24627\t2.655\t0.961\t518\n24631\t3.712\t0.709\t3\n24632\t3.745\t0.575\t6\n24633\t3.382\t0.306\t128\n24634\t3.417\t0.336\t41\n24635\t3.236\t0.278\t27\n24636\t3.365\t0.272\t72\n24638\t3.236\t0.316\t52\n24639\t3.097\t0.0\t2\n24640\t2.718\t0.258\t39\n24641\t3.353\t0.103\t6\n24642\t3.087\t0.178\t51\n24643\t2.988\t0.348\t31\n24644\t2.995\t0.182\t58\n24645\t3.221\t0.798\t58\n24646\t3.104\t0.353\t86\n24647\t3.019\t0.515\t320\n24648\t2.735\t0.176\t341\n24649\t2.703\t0.204\t299\n24650\t2.595\t0.331\t829\n24651\t2.536\t0.58\t227\n24652\t2.472\t0.25\t206\n24653\t2.534\t0.168\t933\n24654\t2.491\t0.326\t1889\n24655\t2.445\t0.168\t254\n24656\t2.51\t0.162\t95\n24657\t3.004\t0.801\t27\n24658\t2.961\t0.516\t71\n24659\t3.105\t0.444\t106\n24666\t3.114\t0.326\t123\n24668\t3.098\t0.303\t230\n28685\t2.353\t0.307\t3006\n28693\t2.684\t0.241\t703\n28694\t2.452\t0.242\t3992\n28695\t2.409\t0.319\t1209\n28696\t2.464\t0.534\t2219\n28697\t2.577\t0.289\t2590\n28698\t2.456\t0.2\t6911\n28699\t2.41\t0.36\t7053\n28700\t2.487\t0.231\t7446\n28701\t2.406\t0.194\t9772\n28702\t2.479\t0.227\t5502\n28703\t2.471\t0.309\t929\n28711\t2.921\t0.239\t1714\n28712\t2.679\t0.232\t2798\n28713\t2.442\t0.272\t911\n28714\t2.445\t0.293\t2412\n28716\t2.475\t0.298\t2904\n28717\t2.428\t0.335\t1207\n28718\t2.458\t0.222\t3289\n28719\t2.487\t0.247\t1033\n28720\t2.81\t0.231\t768\n28721\t2.785\t0.35\t616\n28722\t2.606\t0.37\t1703\n28723\t2.599\t0.29\t531\n28729\t3.125\t0.329\t626\n28730\t3.082\t0.337\t728\n28731\t3.179\t0.249\t153\n28732\t3.022\t0.208\t458\n28734\t3.032\t0.245\t890\n28735\t3.194\t0.258\t616\n28736\t3.148\t0.29\t455\n28737\t3.124\t0.266\t301\n28738\t3.028\t0.263\t602\n28739\t2.987\t0.306\t159\n28740\t2.919\t0.28\t496\n28741\t3.182\t0.365\t119\n28742\t2.892\t0.229\t1179\n28743\t2.821\t0.212\t549\n28744\t2.607\t0.225\t683\n28745\t2.503\t0.227\t1194\n28746\t2.395\t0.326\t1012\n28747\t2.447\t0.267\t941\n28748\t2.383\t0.233\t326\n28749\t2.511\t0.279\t970\n28750\t2.457\t0.24\t1884\n28751\t2.393\t0.172\t533\n28752\t2.849\t0.285\t194\n28753\t2.973\t0.272\t62\n28754\t2.876\t0.236\t326\n28755\t2.753\t0.232\t424\n28762\t3.088\t0.249\t588\n28764\t2.972\t0.34\t2780\n28765\t3.016\t0.335\t10\n28766\t3.297\t0.204\t7\n32781\t2.145\t0.448\t7124\n32789\t2.625\t0.845\t799\n32790\t2.338\t0.282\t13095\n32791\t2.352\t0.77\t18035\n32792\t2.299\t0.427\t5228\n32793\t2.387\t0.292\t15567\n32794\t2.358\t0.225\t12153\n32795\t2.38\t0.29\t13623\n32796\t2.421\t0.301\t14996\n32797\t2.272\t0.223\t19019\n32798\t2.378\t0.357\t12982\n32799\t2.317\t0.208\t831\n32807\t2.796\t0.315\t3153\n32808\t2.497\t0.249\t4631\n32809\t2.375\t0.365\t1523\n32810\t2.347\t0.467\t31965\n32812\t2.419\t0.158\t3311\n32813\t2.436\t0.193\t2109\n32814\t2.4\t0.154\t2649\n32815\t2.661\t0.256\t865\n32816\t2.694\t0.218\t2847\n32817\t2.616\t0.276\t949\n32818\t2.579\t0.438\t4702\n32819\t2.309\t0.553\t1993\n32825\t2.994\t0.343\t2113\n32826\t2.856\t0.3\t2621\n32827\t2.93\t0.228\t1218\n32828\t2.905\t0.207\t2518\n32830\t2.869\t0.237\t2353\n32831\t2.827\t0.209\t3697\n32832\t2.834\t0.197\t5222\n32833\t2.78\t0.177\t4383\n32834\t2.738\t0.205\t8482\n32835\t2.747\t0.193\t1707\n32836\t2.742\t0.193\t2196\n32837\t2.798\t0.21\t502\n32838\t2.734\t0.235\t2716\n32839\t2.767\t0.212\t885\n32840\t2.498\t0.29\t774\n32841\t2.373\t0.261\t1221\n32842\t2.285\t0.417\t9618\n32843\t2.218\t0.348\t3304\n32844\t2.411\t0.2\t303\n32845\t2.462\t0.304\t1297\n32846\t2.393\t0.203\t1953\n32847\t2.305\t0.24\t286\n32848\t2.736\t0.309\t162\n32849\t2.822\t0.336\t202\n32850\t2.624\t0.233\t1404\n32851\t2.579\t0.268\t1906\n32858\t2.759\t0.291\t759\n32860\t2.611\t0.412\t7083\n32861\t2.503\t0.425\t287\n32862\t2.802\t0.399\t255\n32863\t2.837\t0.483\t20\n32864\t3.097\t0.448\t5\n32865\t2.94\t0.02\t3\n32866\t2.916\t0.021\t3\n36877\t1.811\t0.461\t1163\n36885\t2.022\t0.295\t129\n36886\t2.262\t0.492\t1106\n36887\t2.576\t0.889\t536\n36888\t1.937\t0.267\t474\n36889\t2.093\t0.73\t296\n36890\t1.929\t0.743\t373\n36891\t2.283\t0.257\t113\n36892\t2.221\t0.196\t116\n36893\t2.345\t0.817\t110\n36894\t2.065\t0.336\t75\n36895\t2.412\t0.358\t76\n36903\t2.125\t0.259\t44\n36904\t2.154\t1.055\t667\n36905\t2.12\t0.459\t502\n36906\t2.031\t0.453\t512\n36908\t2.395\t0.388\t43\n36909\t2.339\t0.338\t54\n36910\t2.361\t0.307\t101\n36911\t2.57\t0.191\t9\n36912\t2.427\t0.207\t26\n36913\t2.059\t1.008\t121\n36914\t2.126\t0.626\t419\n36915\t2.193\t0.262\t8674\n36921\t2.731\t0.286\t10\n36922\t2.995\t1.207\t51\n36923\t2.406\t0.044\t8\n36924\t2.742\t0.246\t33\n36926\t2.795\t0.182\t52\n36927\t2.799\t0.303\t24\n36928\t2.454\t0.018\t4\n36929\t1.96\t0.0\t6\n36930\t2.434\t0.192\t19\n36931\t2.857\t-1.0\t1\n36932\t2.747\t0.211\t25\n36934\t2.687\t0.139\t19\n36935\t2.806\t0.003\t4\n36936\t2.107\t0.444\t90\n36937\t2.149\t0.225\t250\n36938\t1.882\t0.247\t232\n36939\t2.0\t0.93\t274\n36940\t2.276\t0.144\t56\n36941\t2.293\t1.188\t69\n36942\t2.356\t0.364\t53\n36943\t1.893\t0.555\t149\n36944\t2.373\t0.0\t2\n36945\t2.015\t0.261\t45\n36946\t2.333\t0.203\t37\n36947\t2.152\t0.196\t94\n36954\t2.666\t0.268\t30\n36955\t2.205\t0.007\t16\n36956\t2.285\t0.379\t153\n36957\t2.26\t0.017\t10\n45067\t3.369\t0.477\t389\n45068\t3.242\t-1.0\t1\n45069\t3.524\t0.425\t2\n45070\t3.558\t0.0\t2\n45075\t3.592\t0.346\t36\n45076\t3.383\t0.133\t22\n45081\t3.221\t0.182\t3\n45082\t2.923\t0.336\t4\n45083\t3.382\t-1.0\t1\n45084\t2.961\t0.097\t3\n45087\t3.136\t-1.0\t1\n45088\t3.161\t0.174\t9\n45093\t3.645\t0.249\t17\n45094\t3.529\t0.167\t10\n45101\t3.618\t-1.0\t1\n45106\t3.243\t0.311\t72\n45107\t3.33\t0.219\t17\n45111\t3.704\t0.248\t22\n45112\t3.585\t0.155\t9\n45134\t3.136\t-1.0\t1\n45136\t3.234\t0.09\t7\n45137\t3.405\t0.105\t12\n45138\t3.441\t0.118\t15\n45139\t3.281\t0.068\t7\n45317\t3.245\t0.017\t2\n45318\t3.261\t0.178\t294\n45573\t2.685\t-1.0\t1\n45574\t3.187\t0.261\t35\n49164\t3.052\t0.123\t490\n49165\t3.401\t-1.0\t1\n49166\t2.887\t0.569\t39\n49171\t4.275\t0.055\t2\n49172\t3.412\t0.019\t8\n49177\t3.408\t-1.0\t1\n49178\t3.072\t0.063\t8\n49181\t3.334\t0.086\t3\n49182\t3.217\t0.061\t2\n49183\t2.836\t0.772\t13\n49184\t2.807\t0.438\t17\n49190\t3.648\t0.307\t28\n49191\t2.187\t0.0\t6\n49194\t2.466\t0.0\t2\n49197\t2.818\t0.099\t91\n49202\t2.927\t0.036\t9\n49203\t2.862\t0.432\t19\n49207\t3.807\t0.0\t2\n49208\t3.659\t0.045\t20\n49209\t3.53\t0.075\t16\n49210\t3.447\t0.003\t12\n49211\t3.506\t0.0\t3\n49212\t3.54\t0.0\t3\n49214\t3.463\t0.0\t3\n49215\t3.48\t0.014\t8\n49216\t3.364\t0.0\t3\n49221\t3.144\t0.0\t12\n49229\t2.7\t0.086\t48\n49230\t3.087\t0.019\t2\n49233\t3.145\t0.004\t2\n49413\t2.826\t0.154\t5\n49414\t2.811\t0.216\t411\n49670\t2.681\t0.238\t49\n53261\t3.147\t0.471\t69\n53262\t3.263\t0.176\t31\n53263\t3.143\t0.346\t218\n53264\t2.983\t0.385\t264\n53265\t2.601\t0.322\t2017\n53268\t3.669\t-1.0\t1\n53272\t2.557\t-1.0\t1\n53273\t2.309\t-1.0\t1\n53274\t2.908\t0.179\t18\n53275\t2.828\t0.234\t10\n53276\t2.655\t0.245\t6\n53277\t2.352\t0.06\t13\n53278\t3.3\t0.187\t3\n53279\t3.414\t0.37\t3\n53280\t2.499\t0.185\t12\n53281\t2.448\t0.051\t41\n53282\t2.875\t0.469\t72\n53283\t2.741\t0.288\t381\n53286\t3.495\t0.045\t15\n53292\t2.999\t-1.0\t1\n53293\t3.023\t0.354\t19\n53294\t2.951\t0.089\t21\n53295\t3.08\t-1.0\t1\n53297\t3.715\t-1.0\t1\n53298\t3.418\t0.0\t2\n53299\t2.655\t0.502\t19\n53300\t2.915\t0.495\t30\n53301\t3.049\t0.382\t137\n53303\t2.804\t1.394\t2\n53304\t3.582\t0.06\t4\n53305\t3.584\t0.347\t3\n53308\t3.56\t0.086\t2\n53322\t3.218\t0.048\t2\n53325\t3.468\t0.243\t2\n53326\t3.202\t0.139\t4\n53331\t3.74\t0.255\t4\n53340\t4.01\t0.006\t2\n53509\t2.605\t0.31\t8\n53510\t2.407\t0.249\t1700\n53511\t2.374\t0.163\t1982\n53512\t2.216\t0.23\t533\n53519\t2.935\t0.196\t5\n53520\t2.843\t0.186\t40\n53766\t2.468\t0.199\t96\n53767\t2.448\t0.135\t44\n53768\t1.978\t0.199\t4\n57363\t4.548\t0.464\t19\n57364\t3.083\t0.417\t13\n57366\t3.671\t0.019\t4\n57367\t3.246\t0.035\t4\n57368\t2.795\t0.193\t8\n57369\t2.811\t0.292\t27\n57370\t2.847\t0.192\t233\n57371\t2.715\t0.262\t77\n57372\t2.749\t0.228\t147\n57373\t2.884\t0.306\t43\n57374\t3.288\t0.273\t42\n57375\t3.561\t0.36\t4\n57382\t3.347\t0.097\t2\n57384\t3.775\t0.176\t18\n57385\t3.436\t0.115\t26\n57386\t3.224\t0.255\t49\n57388\t2.956\t0.338\t151\n57389\t2.78\t0.227\t106\n57390\t2.921\t0.275\t97\n57391\t3.357\t0.302\t20\n57393\t3.19\t0.085\t14\n57394\t3.662\t0.527\t210\n57395\t3.917\t0.191\t58\n57399\t3.91\t0.0\t4\n57400\t3.327\t0.062\t5\n57402\t3.01\t0.585\t8\n57406\t4.511\t0.25\t17\n57407\t4.503\t0.08\t6\n57409\t3.926\t0.003\t2\n57411\t3.905\t0.012\t2\n57414\t4.362\t0.205\t19\n57416\t3.87\t0.205\t13\n57417\t3.437\t0.24\t5\n57418\t3.085\t0.21\t66\n57419\t2.96\t0.12\t14\n57420\t3.028\t0.125\t7\n57421\t2.797\t0.286\t150\n57422\t2.978\t0.215\t175\n57423\t2.761\t0.465\t12\n57424\t3.53\t0.211\t22\n57426\t3.874\t0.316\t17\n57427\t4.031\t0.321\t9\n61461\t3.38\t0.299\t100\n61462\t3.21\t0.369\t276\n61463\t3.131\t0.207\t110\n61464\t2.873\t0.244\t730\n61465\t2.696\t0.32\t774\n61466\t2.697\t0.243\t4802\n61467\t2.679\t0.235\t3056\n61468\t2.676\t0.196\t4702\n61469\t2.64\t0.179\t5253\n61470\t2.912\t0.244\t119\n61471\t3.023\t0.242\t152\n61479\t3.617\t0.27\t57\n61480\t3.41\t0.235\t240\n61481\t3.284\t0.297\t129\n61482\t3.002\t0.261\t2058\n61484\t2.758\t0.203\t6616\n61485\t2.714\t0.205\t4052\n61486\t2.713\t0.193\t6710\n61487\t2.863\t0.16\t1964\n61488\t3.199\t0.271\t42\n61489\t3.285\t0.255\t93\n61490\t3.243\t0.31\t221\n61491\t3.183\t0.277\t77\n61497\t3.201\t0.305\t27\n61498\t2.934\t0.476\t9\n61500\t3.451\t0.1\t14\n61502\t3.794\t0.117\t26\n61506\t3.579\t0.083\t13\n61507\t3.584\t0.109\t8\n61508\t3.774\t0.0\t2\n61510\t3.577\t0.217\t22\n61511\t3.739\t0.336\t7\n61512\t3.408\t0.222\t72\n61513\t3.244\t0.257\t147\n61514\t3.003\t0.256\t1665\n61515\t2.906\t0.239\t1623\n61516\t2.798\t0.241\t1057\n61517\t2.754\t0.215\t2863\n61518\t2.714\t0.201\t6668\n61519\t2.668\t0.172\t4216\n61520\t2.879\t0.203\t76\n61521\t3.471\t0.204\t33\n61522\t3.487\t0.164\t39\n61523\t3.141\t0.234\t30\n61530\t3.726\t0.298\t60\n61532\t3.891\t0.328\t80\n65557\t3.412\t0.368\t35\n65558\t2.901\t0.267\t387\n65559\t2.852\t0.348\t489\n65560\t2.881\t0.289\t268\n65561\t2.726\t0.281\t883\n65562\t2.718\t0.226\t7921\n65563\t2.724\t0.272\t1489\n65564\t2.606\t0.234\t5975\n65565\t2.755\t0.225\t5038\n65566\t2.788\t0.225\t1360\n65567\t2.767\t0.214\t227\n65575\t3.418\t0.184\t159\n65576\t3.376\t0.277\t123\n65577\t2.954\t0.528\t373\n65578\t2.911\t0.288\t3355\n65580\t2.816\t0.192\t1907\n65581\t2.839\t0.173\t765\n65582\t2.775\t0.192\t2485\n65583\t3.093\t0.232\t4438\n65584\t3.087\t0.238\t609\n65585\t3.111\t0.393\t484\n65586\t2.949\t0.477\t1247\n65587\t2.995\t0.415\t696\n65593\t3.611\t0.276\t65\n65594\t3.5\t0.164\t59\n65595\t3.551\t0.145\t18\n65596\t3.459\t0.169\t45\n65598\t3.453\t0.18\t175\n65599\t3.683\t0.232\t44\n65600\t3.412\t0.05\t10\n65601\t3.114\t0.282\t6\n65602\t3.516\t0.333\t73\n65603\t3.018\t0.572\t5\n65604\t3.488\t0.196\t81\n65605\t3.266\t0.272\t10\n65606\t3.254\t0.188\t179\n65607\t3.336\t0.194\t34\n65608\t3.554\t0.326\t21\n65609\t2.461\t0.31\t329\n65610\t2.919\t0.288\t2082\n65611\t2.8\t0.267\t1996\n65612\t2.475\t0.252\t240\n65613\t2.833\t0.153\t655\n65614\t2.73\t0.196\t2208\n65615\t2.776\t0.277\t3697\n65616\t2.944\t0.23\t538\n65617\t3.031\t0.277\t17\n65618\t3.147\t0.26\t222\n65619\t3.099\t0.284\t350\n65626\t3.319\t0.35\t170\n65628\t3.382\t0.265\t290\n65629\t3.399\t0.077\t8\n65630\t2.931\t0.387\t17\n65631\t3.438\t0.046\t3\n65632\t3.441\t0.047\t3\n69653\t3.008\t0.308\t233\n69654\t2.799\t0.246\t2597\n69655\t2.789\t0.215\t490\n69656\t2.8\t0.199\t1394\n69657\t2.875\t0.267\t1182\n69658\t2.712\t0.238\t3474\n69659\t2.694\t0.227\t2713\n69660\t2.758\t0.28\t2082\n69661\t2.701\t0.25\t5780\n69662\t2.642\t0.425\t3822\n69663\t2.661\t0.254\t1485\n69671\t3.218\t0.274\t409\n69672\t2.991\t0.248\t1900\n69673\t2.903\t0.356\t1449\n69674\t2.907\t0.475\t2267\n69676\t2.885\t0.203\t5377\n69677\t2.863\t0.235\t2604\n69678\t2.776\t0.213\t6900\n69679\t3.032\t0.245\t617\n69680\t2.974\t0.221\t1026\n69681\t2.957\t0.377\t932\n69682\t2.928\t0.391\t2452\n69683\t2.872\t0.46\t1745\n69689\t3.393\t0.345\t120\n69690\t3.542\t0.471\t133\n69691\t3.384\t0.328\t54\n69692\t3.319\t0.289\t122\n69694\t3.354\t0.248\t151\n69695\t3.179\t0.349\t113\n69696\t3.251\t0.372\t116\n69697\t3.069\t0.3\t85\n69698\t3.114\t0.336\t295\n69699\t3.273\t0.344\t35\n69700\t3.214\t0.336\t160\n69701\t2.947\t0.43\t25\n69702\t3.224\t0.207\t183\n69703\t3.182\t0.198\t73\n69704\t2.972\t0.302\t357\n69705\t2.911\t0.327\t1000\n69706\t2.885\t0.369\t1787\n69707\t2.801\t0.257\t2221\n69708\t2.851\t0.255\t681\n69709\t2.879\t0.222\t2186\n69710\t2.757\t0.265\t4336\n69711\t2.675\t0.201\t2687\n69712\t2.804\t0.27\t1229\n69713\t2.554\t0.473\t192\n69714\t3.224\t0.457\t625\n69715\t3.107\t0.371\t1402\n69722\t3.417\t0.306\t157\n69723\t2.64\t0.0\t4\n69724\t3.21\t0.343\t893\n69725\t3.181\t0.284\t122\n69726\t3.088\t0.296\t154\n69727\t2.915\t-1.0\t1\n69728\t2.915\t-1.0\t1\n69729\t2.465\t0.0\t6\n77843\t3.904\t0.444\t476\n77844\t3.931\t0.219\t13\n77847\t3.213\t0.001\t2\n77848\t3.318\t0.114\t5\n77849\t3.574\t0.313\t3\n77850\t3.205\t-1.0\t1\n77851\t3.463\t0.411\t4\n77852\t3.491\t0.534\t2\n77853\t3.822\t0.039\t3\n77854\t3.511\t0.765\t6\n77856\t3.502\t0.647\t78\n77861\t3.9\t0.078\t12\n77862\t3.806\t0.091\t8\n77864\t3.807\t-1.0\t1\n77865\t3.535\t0.137\t7\n77870\t3.386\t0.218\t17\n77871\t3.245\t0.006\t2\n77874\t3.754\t0.595\t67\n77875\t3.655\t0.42\t25\n77879\t3.899\t0.365\t31\n77880\t4.729\t0.676\t2\n77898\t4.496\t0.606\t3\n77902\t3.48\t0.0\t4\n77905\t3.736\t0.08\t11\n77906\t3.724\t0.099\t26\n77907\t3.562\t0.045\t4\n77917\t4.592\t-1.0\t1\n78085\t3.76\t0.113\t8\n78086\t3.713\t0.29\t1287\n78342\t3.557\t0.28\t73\n81940\t3.661\t0.289\t176\n81943\t3.019\t-1.0\t1\n81944\t2.85\t-1.0\t1\n81945\t3.324\t0.0\t2\n81951\t3.151\t-1.0\t1\n81952\t3.185\t0.116\t25\n81957\t3.75\t0.247\t3\n81958\t3.579\t0.106\t3\n81967\t3.517\t0.128\t5\n81969\t3.26\t0.128\t3\n81970\t3.346\t0.13\t71\n81971\t3.246\t0.132\t128\n81975\t3.895\t0.219\t5\n81976\t3.719\t0.202\t10\n81997\t3.133\t0.007\t2\n81999\t3.088\t0.086\t20\n82000\t3.26\t0.098\t4\n82001\t3.332\t-1.0\t1\n82002\t3.427\t0.083\t10\n82003\t3.25\t0.171\t45\n82012\t3.805\t-1.0\t1\n82182\t3.212\t0.212\t411\n82438\t2.973\t0.227\t13\n86044\t2.357\t-1.0\t1\n86050\t3.301\t0.106\t5\n86051\t3.223\t0.408\t15\n86053\t3.906\t-1.0\t1\n86062\t2.711\t0.034\t2\n86068\t3.534\t0.111\t4\n86069\t3.014\t0.266\t86\n86277\t3.132\t0.206\t3\n86278\t3.08\t0.205\t1002\n86279\t2.704\t0.168\t460\n86280\t2.561\t0.163\t44\n86287\t3.292\t0.078\t26\n86288\t3.128\t0.068\t7\n86534\t2.902\t0.101\t13\n86535\t2.446\t0.18\t5\n90134\t3.158\t0.093\t9\n90138\t2.576\t0.185\t3\n90139\t2.835\t0.161\t13\n90140\t2.925\t0.174\t8\n90141\t3.761\t-1.0\t1\n90143\t3.058\t0.0\t2\n90144\t2.522\t0.784\t3\n90145\t2.482\t0.209\t46\n90146\t3.164\t0.558\t44\n90147\t2.774\t0.281\t113\n90150\t3.233\t0.034\t7\n90156\t3.193\t-1.0\t1\n90157\t2.921\t-1.0\t1\n90161\t3.271\t0.0\t2\n90162\t3.869\t0.023\t3\n90163\t3.573\t0.143\t28\n90164\t3.347\t0.022\t18\n90165\t2.62\t0.414\t54\n90168\t3.427\t0.036\t8\n90194\t3.897\t-1.0\t1\n90373\t3.171\t0.268\t3\n90374\t2.926\t0.185\t13105\n90375\t2.587\t0.178\t1875\n90376\t2.456\t0.185\t917\n90383\t3.113\t0.322\t22\n90384\t3.044\t0.167\t14\n90630\t2.624\t0.183\t73\n90631\t2.431\t0.217\t59\n90632\t2.541\t0.214\t13\n94231\t2.269\t0.138\t9\n94234\t2.503\t0.095\t6\n94235\t2.637\t0.592\t3\n94236\t3.236\t0.203\t2\n94242\t2.529\t0.487\t29\n94243\t2.969\t0.254\t17\n94264\t3.486\t-1.0\t1\n94286\t2.784\t0.083\t10\n94470\t2.664\t0.196\t1656\n94471\t2.46\t0.141\t1730\n94472\t2.373\t0.21\t563\n94479\t3.002\t0.084\t9\n94480\t2.809\t0.211\t72\n94726\t2.339\t0.204\t93\n94727\t2.461\t0.173\t28\n94728\t2.22\t0.033\t36\n98328\t2.585\t0.33\t139\n98329\t3.026\t0.483\t6\n98330\t3.053\t0.116\t7\n98331\t2.861\t0.157\t7\n98332\t3.016\t0.06\t2\n98333\t2.906\t-1.0\t1\n98334\t3.171\t-1.0\t1\n98335\t2.778\t0.078\t6\n98336\t2.83\t0.006\t4\n98337\t2.984\t0.301\t63\n98338\t3.557\t0.536\t97\n98339\t2.852\t0.31\t66\n98341\t3.47\t0.104\t3\n98349\t3.572\t-1.0\t1\n98350\t3.259\t0.092\t4\n98354\t3.11\t0.137\t19\n98355\t3.017\t0.193\t20\n98356\t3.203\t0.219\t43\n98357\t3.129\t0.376\t67\n98378\t3.04\t-1.0\t1\n98381\t3.568\t-1.0\t1\n98382\t3.15\t0.188\t5\n98383\t3.133\t0.037\t3\n98384\t3.045\t0.0\t2\n98386\t3.121\t0.097\t12\n98387\t3.325\t0.016\t2\n98565\t2.688\t0.12\t18\n98566\t2.619\t0.153\t6878\n98567\t2.404\t0.141\t2515\n98568\t2.41\t0.174\t638\n98575\t3.098\t0.146\t259\n98576\t2.876\t0.25\t37\n98822\t2.158\t0.152\t4143\n98823\t2.42\t0.131\t56\n98824\t2.468\t0.236\t5\n102425\t3.087\t0.258\t19\n102426\t2.867\t0.103\t4\n102427\t3.03\t0.131\t2\n102428\t2.886\t-1.0\t1\n102431\t3.326\t0.375\t7\n102432\t2.775\t0.168\t35\n102433\t2.737\t0.092\t14\n102434\t3.052\t0.275\t320\n102435\t2.947\t0.279\t501\n102438\t3.136\t0.442\t2\n102442\t3.198\t-1.0\t1\n102444\t2.998\t0.081\t3\n102445\t2.848\t0.318\t2\n102447\t3.116\t0.312\t4\n102449\t2.668\t0.007\t12\n102450\t2.65\t0.156\t37\n102451\t2.804\t0.128\t97\n102452\t3.71\t0.389\t143\n102453\t3.196\t0.352\t77\n102476\t2.993\t0.038\t4\n102478\t3.256\t-1.0\t1\n102479\t3.089\t0.272\t5\n102480\t2.61\t0.003\t4\n102481\t3.397\t-1.0\t1\n102482\t2.754\t0.035\t4\n102483\t2.878\t0.096\t25\n102661\t2.718\t0.111\t3\n102662\t2.482\t0.226\t2788\n102663\t2.511\t0.177\t8370\n102664\t2.493\t0.179\t2602\n102671\t2.643\t0.184\t45\n102672\t2.962\t0.176\t53\n102918\t2.052\t0.149\t4796\n102919\t2.51\t0.179\t433\n102920\t2.374\t0.161\t19\n106522\t2.941\t0.371\t219\n106523\t2.892\t0.261\t5\n106524\t2.877\t0.111\t7\n106525\t2.863\t0.149\t85\n106526\t2.98\t0.096\t34\n106527\t2.834\t0.242\t89\n106528\t2.743\t0.335\t128\n106529\t2.701\t0.254\t143\n106530\t2.739\t0.223\t715\n106531\t2.904\t0.242\t866\n106536\t3.002\t0.349\t8\n106537\t2.799\t0.229\t2\n106538\t3.086\t0.144\t4\n106540\t2.735\t0.302\t14\n106541\t2.768\t0.387\t2\n106542\t3.159\t0.157\t17\n106543\t2.781\t0.205\t11\n106545\t3.031\t0.266\t33\n106546\t3.013\t0.266\t96\n106547\t2.844\t0.229\t38\n106548\t2.92\t0.185\t202\n106549\t3.146\t0.267\t200\n106551\t3.701\t0.004\t2\n106552\t3.508\t0.08\t2\n106554\t3.692\t-1.0\t1\n106556\t3.882\t-1.0\t1\n106566\t3.665\t0.215\t7\n106570\t3.257\t0.114\t4\n106571\t2.863\t0.015\t4\n106572\t2.976\t0.164\t5\n106574\t3.224\t0.085\t11\n106575\t2.779\t0.247\t4\n106576\t2.981\t0.124\t18\n106577\t3.014\t0.072\t10\n106579\t3.124\t0.117\t28\n106757\t2.221\t0.408\t15\n106758\t2.401\t0.141\t81624\n106759\t2.351\t0.184\t21226\n106760\t2.431\t0.13\t2185\n106767\t2.782\t0.218\t230\n106768\t2.742\t0.183\t221\n107014\t2.077\t0.157\t13871\n107015\t2.286\t0.144\t1126\n107016\t2.261\t0.186\t6\n107023\t2.555\t-1.0\t1\n110619\t2.657\t0.367\t240\n110620\t2.816\t0.234\t12\n110621\t2.849\t0.093\t2\n110622\t2.914\t0.068\t12\n110623\t2.683\t0.337\t23\n110624\t2.349\t0.547\t34\n110625\t2.914\t0.287\t44\n110626\t2.37\t0.334\t254\n110627\t2.823\t0.245\t593\n110629\t3.216\t-1.0\t1\n110632\t3.168\t0.262\t18\n110633\t3.517\t-1.0\t1\n110634\t3.188\t-1.0\t1\n110636\t2.935\t0.025\t5\n110637\t2.659\t0.073\t4\n110638\t2.964\t0.14\t43\n110641\t2.808\t0.245\t6\n110642\t2.669\t0.36\t18\n110643\t2.899\t0.004\t2\n110644\t2.907\t0.266\t28\n110645\t3.111\t0.233\t239\n110654\t3.746\t0.042\t4\n110664\t3.213\t0.095\t2\n110665\t3.535\t-1.0\t1\n110666\t3.211\t-1.0\t1\n110669\t2.652\t-1.0\t1\n110670\t3.108\t0.335\t20\n110671\t3.078\t0.114\t48\n110672\t2.505\t0.007\t4\n110673\t2.66\t0.009\t3\n110674\t2.75\t0.014\t4\n110675\t2.821\t0.241\t50\n110682\t3.786\t-1.0\t1\n110684\t3.602\t0.211\t2\n110853\t2.51\t0.032\t24\n110854\t2.424\t0.18\t7855\n110855\t2.35\t0.165\t15280\n110856\t2.397\t0.151\t2600\n110863\t2.753\t0.315\t114\n110864\t2.718\t0.212\t256\n111110\t2.057\t0.162\t5916\n111111\t2.347\t0.183\t580\n111120\t3.119\t-1.0\t1\n114716\t2.861\t0.204\t423\n114717\t2.956\t0.144\t62\n114718\t3.07\t0.105\t55\n114719\t2.884\t0.235\t61\n114720\t2.67\t0.388\t26\n114721\t2.959\t0.219\t18\n114722\t2.832\t0.233\t173\n114723\t2.869\t0.229\t1269\n114727\t2.635\t0.109\t4\n114728\t2.753\t-1.0\t1\n114730\t2.796\t0.0\t2\n114733\t2.529\t0.017\t3\n114737\t2.923\t0.325\t25\n114738\t3.191\t0.145\t154\n114739\t3.085\t0.122\t9\n114740\t3.439\t0.421\t52\n114741\t3.061\t0.179\t190\n114743\t3.288\t-1.0\t1\n114745\t2.617\t0.062\t2\n114759\t3.289\t0.325\t3\n114760\t2.798\t-1.0\t1\n114762\t2.961\t-1.0\t1\n114765\t2.702\t-1.0\t1\n114766\t3.089\t0.135\t10\n114767\t3.047\t0.066\t62\n114768\t3.431\t0.011\t4\n114769\t3.198\t-1.0\t1\n114770\t3.331\t0.009\t16\n114771\t3.287\t0.091\t62\n114780\t3.048\t-1.0\t1\n114949\t2.438\t0.289\t13\n114950\t2.447\t0.207\t5143\n114951\t2.359\t0.153\t15318\n114952\t2.422\t0.153\t2635\n114959\t2.635\t0.231\t43\n114960\t2.669\t0.201\t795\n115206\t2.167\t0.159\t1651\n115207\t2.384\t0.147\t685\n115208\t2.377\t-1.0\t1\n118813\t2.913\t0.463\t3054\n118814\t2.839\t0.079\t11\n118815\t2.711\t0.001\t4\n118816\t2.916\t0.195\t36\n118817\t2.906\t0.257\t48\n118818\t3.003\t0.19\t1185\n118819\t2.89\t0.233\t2162\n118821\t4.015\t-1.0\t1\n118826\t2.96\t-1.0\t1\n118828\t2.952\t0.11\t39\n118829\t2.507\t0.024\t3\n118830\t2.938\t0.19\t22\n118831\t2.849\t0.2\t37\n118834\t2.765\t0.134\t305\n118835\t3.108\t0.155\t31\n118836\t3.39\t0.219\t189\n118837\t3.139\t0.235\t3644\n118839\t3.839\t-1.0\t1\n118858\t2.994\t-1.0\t1\n118861\t2.937\t0.024\t4\n118862\t2.749\t0.231\t11\n118863\t3.089\t0.14\t128\n118864\t2.73\t0.419\t10\n118867\t4.236\t-1.0\t1\n119046\t2.301\t0.205\t1784\n119047\t2.304\t0.132\t28601\n119048\t2.278\t0.16\t3943\n119055\t2.752\t0.18\t205\n119056\t2.764\t0.209\t1501\n119302\t2.425\t0.207\t1326\n119303\t2.302\t0.163\t1333\n119304\t2.514\t0.012\t2\n122910\t2.933\t0.511\t46\n122911\t2.702\t0.368\t7\n122912\t2.663\t0.513\t47\n122913\t2.561\t0.197\t22\n122914\t2.985\t0.238\t269\n122915\t2.799\t0.235\t1020\n122917\t3.448\t-1.0\t1\n122920\t3.428\t0.0\t2\n122922\t3.262\t0.063\t13\n122924\t3.141\t0.257\t36\n122925\t3.024\t0.038\t68\n122926\t3.08\t0.273\t30\n122929\t3.163\t0.018\t2\n122930\t2.751\t0.402\t11\n122932\t3.359\t0.441\t18\n122933\t3.021\t0.24\t382\n122935\t3.714\t-1.0\t1\n122955\t3.014\t0.026\t6\n122957\t3.238\t0.02\t4\n122958\t3.092\t0.105\t29\n122960\t3.074\t0.0\t2\n122961\t2.928\t0.007\t10\n122962\t3.518\t0.023\t8\n122963\t2.553\t0.0\t2\n123142\t2.737\t0.301\t850\n123143\t2.41\t0.137\t13630\n123144\t2.395\t0.157\t2260\n123151\t2.989\t0.176\t11\n123152\t2.86\t0.169\t504\n123398\t2.392\t0.233\t92\n123399\t2.348\t0.155\t241\n127007\t3.02\t0.246\t86\n127008\t2.974\t0.088\t8\n127009\t3.494\t0.451\t21\n127010\t2.829\t0.38\t37\n127011\t2.708\t0.245\t196\n127014\t3.469\t0.079\t7\n127016\t3.23\t0.0\t2\n127018\t3.049\t0.131\t23\n127020\t2.953\t0.101\t22\n127021\t3.009\t0.238\t45\n127022\t3.086\t0.158\t56\n127023\t3.058\t0.009\t4\n127025\t3.335\t0.196\t5\n127026\t3.283\t0.062\t7\n127027\t2.651\t0.769\t24\n127028\t3.434\t0.203\t18\n127029\t3.123\t0.373\t261\n127031\t3.789\t0.084\t6\n127032\t4.457\t-1.0\t1\n127033\t3.667\t-1.0\t1\n127036\t4.074\t0.0\t2\n127039\t4.048\t-1.0\t1\n127046\t4.093\t-1.0\t1\n127050\t2.953\t0.101\t7\n127052\t3.076\t0.218\t4\n127053\t3.176\t0.156\t3\n127054\t2.924\t0.124\t38\n127055\t2.909\t0.082\t10\n127059\t4.205\t0.525\t4\n127068\t3.948\t0.013\t2\n127237\t2.513\t0.281\t7\n127238\t2.422\t0.173\t417\n127239\t2.418\t0.176\t782\n127240\t2.39\t0.178\t153\n127247\t2.978\t0.075\t57\n127248\t2.775\t0.184\t61\n127494\t2.47\t0.44\t42\n127495\t2.406\t0.044\t2\n131109\t3.618\t0.424\t54\n131110\t3.321\t0.053\t36\n131111\t2.973\t0.025\t4\n131112\t3.771\t0.94\t5\n131113\t3.619\t0.341\t9\n131114\t3.143\t0.38\t30\n131116\t2.916\t0.159\t157\n131117\t2.917\t0.159\t121\n131118\t3.04\t0.268\t107\n131119\t2.922\t0.222\t15\n131120\t3.307\t0.038\t4\n131122\t3.411\t0.477\t35\n131123\t3.351\t-1.0\t1\n131127\t3.814\t0.114\t37\n131128\t3.49\t0.107\t29\n131130\t2.988\t0.0\t6\n131137\t2.864\t-1.0\t1\n131138\t3.615\t0.013\t2\n131144\t3.909\t0.169\t5\n131146\t3.079\t0.227\t59\n131147\t2.946\t0.309\t17\n131148\t2.938\t0.152\t48\n131149\t2.889\t0.075\t105\n131150\t2.932\t0.142\t86\n131151\t2.965\t0.315\t19\n131154\t4.158\t0.0\t3\n131155\t3.469\t0.0\t2\n135208\t3.392\t0.193\t9\n135209\t2.553\t0.506\t113\n135210\t2.846\t0.341\t42\n135212\t2.866\t0.212\t135\n135213\t2.927\t0.123\t16\n135214\t2.871\t0.163\t78\n135215\t2.95\t0.206\t41\n135216\t2.799\t0.488\t10\n135217\t2.858\t0.374\t47\n135218\t3.171\t0.414\t88\n135219\t3.525\t0.292\t6\n135226\t2.999\t0.0\t6\n135230\t3.926\t0.069\t11\n135234\t3.555\t0.069\t7\n135237\t4.014\t0.007\t2\n135240\t2.589\t0.032\t8\n135241\t2.464\t0.036\t10\n135242\t2.986\t0.141\t77\n135243\t2.924\t-1.0\t1\n135244\t2.773\t0.037\t5\n135245\t3.046\t0.0\t2\n135246\t2.825\t0.185\t175\n135247\t2.784\t0.162\t65\n135248\t3.475\t0.422\t13\n135249\t3.243\t0.492\t4\n135250\t3.627\t0.385\t15\n135251\t3.822\t0.36\t16\n135258\t3.001\t0.0\t2\n135260\t3.608\t0.228\t9\n135430\t2.312\t0.051\t4\n139303\t3.535\t0.292\t31\n139304\t3.38\t0.183\t24\n139305\t2.623\t0.529\t104\n139306\t3.083\t0.33\t321\n139308\t2.96\t0.162\t229\n139309\t2.923\t0.274\t63\n139310\t2.882\t0.149\t259\n139311\t3.27\t0.202\t438\n139312\t2.668\t0.417\t260\n139313\t3.241\t0.429\t63\n139314\t3.163\t0.399\t546\n139315\t3.191\t0.461\t290\n139321\t3.855\t0.226\t50\n139322\t3.709\t0.138\t100\n139323\t3.684\t0.123\t46\n139324\t3.633\t0.072\t70\n139326\t3.545\t0.123\t112\n139327\t3.991\t0.038\t4\n139328\t3.786\t0.526\t8\n139329\t3.979\t0.001\t4\n139330\t3.537\t0.014\t6\n139331\t3.355\t0.108\t12\n139332\t3.159\t0.173\t21\n139333\t3.624\t0.136\t27\n139334\t3.334\t0.167\t126\n139336\t3.239\t0.487\t10\n139337\t2.614\t0.35\t62\n139338\t3.123\t0.251\t218\n139339\t3.229\t0.309\t1334\n139340\t3.562\t0.717\t14\n139341\t2.945\t0.19\t49\n139342\t2.914\t0.165\t139\n139343\t2.866\t0.183\t165\n139344\t3.14\t0.274\t105\n139345\t3.078\t0.295\t50\n139346\t3.558\t0.371\t43\n139347\t3.512\t0.445\t86\n139354\t3.623\t0.348\t57\n139356\t3.581\t0.414\t207\n139357\t3.553\t0.26\t12\n139358\t3.738\t0.14\t26\n139526\t2.278\t0.056\t6\n143399\t2.881\t0.269\t29\n143400\t3.289\t0.207\t165\n143401\t2.903\t0.415\t136\n143402\t2.928\t0.263\t822\n143404\t2.981\t0.274\t225\n143405\t3.002\t0.373\t123\n143406\t2.929\t0.211\t1350\n143407\t3.15\t0.245\t392\n143408\t3.115\t0.214\t390\n143409\t2.892\t0.454\t277\n143410\t3.152\t0.327\t288\n143411\t3.118\t0.344\t421\n143417\t3.489\t0.306\t26\n143418\t3.66\t0.441\t102\n143419\t2.889\t0.02\t3\n143420\t3.311\t0.239\t14\n143422\t3.519\t0.117\t10\n143423\t3.405\t0.182\t14\n143424\t3.622\t0.201\t10\n143425\t2.902\t0.279\t14\n143426\t3.441\t0.397\t31\n143427\t2.869\t0.248\t16\n143428\t2.769\t0.111\t12\n143430\t3.342\t0.101\t8\n143433\t3.029\t0.413\t218\n143434\t2.63\t0.358\t237\n143435\t2.974\t0.223\t787\n143436\t2.871\t0.256\t91\n143437\t3.02\t0.255\t120\n143438\t2.97\t0.262\t730\n143439\t2.839\t0.219\t570\n143440\t2.939\t0.303\t600\n143441\t2.709\t0.226\t130\n143442\t3.393\t0.358\t470\n143443\t3.237\t0.326\t869\n143450\t3.569\t0.067\t21\n143452\t3.3\t0.463\t144\n143453\t2.771\t0.0\t6\n143454\t3.655\t0.021\t3\n143455\t2.882\t0.023\t3\n151589\t4.027\t0.433\t171\n151590\t3.818\t0.355\t2\n151594\t3.871\t0.0\t2\n151598\t3.586\t0.162\t20\n151599\t3.353\t0.049\t2\n151601\t3.937\t0.0\t2\n151602\t3.839\t0.296\t15\n151603\t3.859\t0.378\t36\n151607\t4.188\t0.029\t2\n151631\t3.626\t0.064\t26\n151633\t3.814\t0.025\t4\n151634\t3.849\t0.071\t10\n151635\t4.057\t0.0\t2\n151814\t3.815\t0.247\t64\n152069\t3.392\t0.0\t2\n152070\t3.728\t0.185\t7\n155686\t3.715\t0.263\t197\n155692\t3.291\t0.0\t2\n155693\t3.275\t0.0\t2\n155695\t3.371\t-1.0\t1\n155696\t3.544\t0.083\t4\n155697\t3.415\t0.077\t36\n155698\t3.446\t0.074\t26\n155699\t3.346\t0.137\t70\n155703\t4.039\t0.114\t2\n155704\t3.785\t-1.0\t1\n155725\t3.3\t0.0\t4\n155726\t3.174\t0.064\t8\n155727\t3.236\t0.051\t4\n155728\t3.411\t0.02\t2\n155729\t3.492\t0.006\t3\n155730\t3.53\t0.066\t10\n155731\t3.424\t0.132\t88\n155740\t3.534\t-1.0\t1\n155910\t3.584\t0.272\t112\n156166\t2.793\t0.019\t4\n159783\t4.713\t0.009\t2\n159790\t3.627\t0.388\t4\n159795\t3.832\t0.039\t12\n159797\t3.719\t0.347\t56\n159799\t4.168\t0.0\t2\n159822\t3.558\t0.003\t2\n160005\t3.476\t0.074\t35\n160006\t3.258\t0.191\t4054\n160007\t2.913\t0.142\t1412\n160008\t2.796\t0.249\t708\n160015\t3.636\t0.026\t3\n160016\t3.367\t0.408\t27\n160262\t2.99\t0.127\t72\n160263\t3.031\t0.264\t24\n163884\t3.712\t0.084\t3\n163885\t3.713\t0.26\t5\n163889\t3.399\t0.0\t2\n163890\t3.989\t0.377\t5\n163892\t3.476\t0.0\t2\n163893\t3.302\t0.424\t87\n163917\t3.364\t0.045\t5\n163918\t3.314\t0.355\t6\n163922\t4.034\t-1.0\t1\n164101\t3.255\t0.125\t9\n164102\t3.029\t0.219\t13290\n164103\t2.812\t0.186\t1007\n164104\t2.599\t0.231\t718\n164111\t3.355\t0.275\t19\n164112\t3.184\t0.119\t22\n164358\t2.805\t0.151\t103\n164359\t2.642\t0.21\t22\n164360\t2.844\t0.216\t9\n167977\t2.746\t0.433\t12\n167986\t3.166\t-1.0\t1\n167988\t3.668\t0.819\t15\n167989\t2.936\t0.216\t20\n168198\t2.975\t0.237\t953\n168199\t2.749\t0.24\t450\n168200\t2.532\t0.296\t44\n168207\t3.165\t0.13\t49\n168208\t3.318\t0.08\t14\n168454\t2.601\t0.169\t111\n168455\t2.727\t0.161\t25\n172074\t2.912\t0.373\t1298\n172076\t3.157\t0.34\t9\n172078\t3.301\t0.025\t5\n172079\t3.222\t0.067\t8\n172082\t3.28\t0.259\t30\n172083\t3.177\t0.165\t15\n172084\t3.327\t0.193\t23\n172085\t3.272\t0.292\t557\n172087\t3.799\t-1.0\t1\n172088\t3.528\t-1.0\t1\n172106\t2.727\t0.462\t6\n172107\t3.735\t-1.0\t1\n172109\t3.252\t0.082\t43\n172110\t3.239\t0.183\t9\n172111\t3.197\t0.132\t22\n172112\t3.584\t0.076\t4\n172114\t3.693\t0.232\t12\n172115\t3.359\t0.28\t40\n172124\t3.744\t0.013\t2\n172293\t3.03\t0.256\t14\n172294\t2.789\t0.167\t9278\n172295\t2.577\t0.18\t3098\n172296\t2.467\t0.259\t1296\n172303\t2.908\t0.239\t150\n172304\t2.957\t0.217\t165\n172550\t2.341\t0.159\t4663\n172551\t2.511\t0.149\t218\n172559\t2.202\t0.279\t10\n180268\t3.063\t0.299\t1102\n180269\t2.945\t0.515\t10\n180270\t3.082\t0.36\t14\n180271\t3.086\t-1.0\t1\n180272\t3.242\t0.131\t2\n180273\t3.439\t0.268\t6\n180274\t3.072\t0.196\t129\n180275\t3.213\t0.211\t24\n180276\t3.091\t0.153\t98\n180277\t3.22\t0.208\t173\n180295\t3.709\t0.109\t4\n180296\t3.674\t0.024\t2\n180298\t3.338\t0.368\t8\n180299\t3.08\t0.02\t2\n180300\t3.051\t0.136\t6\n180301\t3.2\t0.237\t31\n180302\t3.182\t0.186\t59\n180303\t2.975\t0.164\t24\n180304\t3.255\t-1.0\t1\n180306\t2.931\t0.041\t12\n180307\t2.734\t0.077\t20\n180485\t2.755\t0.204\t14\n180486\t2.611\t0.146\t28221\n180487\t2.363\t0.139\t11820\n180488\t2.408\t0.153\t1326\n180495\t2.75\t0.202\t77\n180496\t2.828\t0.175\t621\n180742\t2.163\t0.16\t10524\n180743\t2.394\t0.13\t880\n180744\t2.447\t0.224\t6\n180751\t2.688\t0.012\t5\n180752\t2.464\t0.284\t5\n184365\t2.871\t0.285\t392\n184366\t3.302\t0.52\t4\n184367\t2.634\t0.029\t2\n184368\t2.701\t-1.0\t1\n184369\t3.216\t0.427\t13\n184370\t3.333\t0.264\t124\n184371\t3.063\t0.086\t29\n184372\t3.343\t0.188\t5\n184373\t3.193\t0.194\t327\n184396\t2.794\t0.148\t26\n184397\t2.962\t0.816\t2\n184398\t2.985\t0.39\t19\n184399\t3.255\t0.176\t61\n184400\t3.215\t0.059\t6\n184402\t3.402\t0.003\t2\n184403\t3.348\t0.247\t21\n184412\t3.283\t0.005\t3\n184581\t2.634\t0.295\t6\n184582\t2.568\t0.171\t12855\n184583\t2.441\t0.153\t3081\n184584\t2.416\t0.166\t1225\n184591\t2.739\t0.094\t61\n184592\t2.805\t0.202\t148\n184837\t2.886\t-1.0\t1\n184838\t2.196\t0.199\t2670\n184839\t2.45\t0.148\t279\n184848\t2.287\t0.009\t6\n188462\t3.197\t0.284\t549\n188463\t3.567\t0.174\t71\n188465\t3.129\t0.275\t2\n188466\t3.378\t0.343\t81\n188467\t3.278\t0.49\t122\n188468\t3.121\t0.318\t140\n188469\t3.118\t0.179\t936\n188472\t3.155\t0.189\t11\n188486\t2.788\t-1.0\t1\n188487\t3.697\t0.043\t2\n188490\t3.361\t-1.0\t1\n188491\t3.319\t0.01\t2\n188492\t3.071\t0.101\t16\n188494\t3.253\t0.286\t17\n188495\t3.308\t0.347\t126\n188496\t3.082\t0.313\t5\n188497\t3.342\t0.379\t39\n188498\t3.685\t0.035\t11\n188499\t2.981\t0.028\t91\n188508\t3.265\t0.257\t3\n188677\t2.545\t0.218\t11\n188678\t2.381\t0.169\t6266\n188679\t2.365\t0.135\t7580\n188680\t2.446\t0.147\t1073\n188687\t2.778\t0.159\t82\n188688\t2.732\t0.177\t615\n188934\t2.405\t0.21\t1341\n188935\t2.38\t0.131\t281\n188936\t2.152\t0.004\t2\n188944\t2.657\t0.082\t9\n192559\t3.554\t0.226\t4083\n192561\t2.716\t-1.0\t1\n192562\t3.443\t0.292\t14\n192563\t3.163\t0.241\t12\n192564\t3.698\t0.229\t62\n192565\t3.522\t0.275\t796\n192572\t3.827\t0.068\t3\n192587\t3.3\t0.096\t6\n192588\t3.119\t0.06\t30\n192589\t3.417\t0.237\t27\n192590\t3.436\t0.273\t279\n192591\t3.326\t0.24\t399\n192592\t2.823\t0.436\t7\n192774\t2.472\t0.2\t1611\n192775\t2.535\t0.173\t5301\n192776\t2.731\t0.17\t305\n192783\t2.898\t0.163\t18\n192784\t3.194\t0.23\t847\n193030\t2.719\t0.257\t1011\n193031\t2.571\t0.187\t351\n196656\t2.962\t0.143\t4\n196659\t2.984\t0.064\t32\n196660\t3.488\t0.2\t53\n196661\t3.212\t0.24\t422\n196664\t3.568\t0.009\t4\n196686\t3.302\t0.12\t162\n196687\t3.375\t0.136\t15\n196690\t3.639\t0.028\t8\n196691\t3.016\t0.052\t32\n196870\t2.546\t0.218\t32\n196871\t2.669\t0.125\t3379\n196872\t2.711\t0.152\t719\n196880\t3.123\t0.225\t349\n197126\t2.828\t0.381\t77\n197127\t2.606\t0.119\t56\n197128\t2.751\t-1.0\t1\n200753\t2.968\t0.291\t109\n200754\t3.781\t0.18\t17\n200755\t2.926\t0.006\t4\n200756\t3.55\t0.394\t50\n200757\t3.215\t0.314\t179\n200759\t3.813\t0.152\t8\n200760\t3.68\t0.079\t15\n200778\t3.2\t0.021\t2\n200779\t2.798\t0.024\t24\n200781\t2.816\t0.38\t2\n200782\t3.146\t0.138\t51\n200783\t3.783\t-1.0\t1\n200784\t3.608\t0.06\t2\n200787\t3.701\t0.063\t13\n200966\t2.737\t0.248\t164\n200967\t2.638\t0.193\t666\n200968\t2.735\t0.196\t128\n200975\t3.61\t0.145\t5\n200976\t2.787\t0.336\t60\n201222\t2.754\t0.18\t12\n201223\t2.897\t0.149\t12\n204850\t3.629\t0.476\t976\n204851\t2.793\t0.054\t34\n204852\t3.367\t0.645\t98\n204853\t3.325\t0.463\t310\n204855\t3.996\t0.344\t33\n204856\t3.682\t0.08\t11\n204857\t3.947\t0.023\t6\n204858\t3.186\t0.0\t6\n204863\t3.44\t0.0\t11\n204866\t3.929\t-1.0\t1\n204870\t3.525\t0.118\t8\n204872\t4.135\t0.14\t3\n204874\t3.301\t0.205\t57\n204875\t3.121\t0.234\t55\n204876\t3.203\t0.168\t70\n204877\t3.061\t0.08\t42\n204878\t3.284\t0.173\t281\n204879\t4.309\t0.589\t23\n204881\t3.669\t0.346\t5\n204882\t3.46\t0.055\t10\n204883\t3.636\t0.704\t18\n204892\t4.537\t-1.0\t1\n205061\t2.753\t0.308\t17\n205062\t2.528\t0.15\t3869\n205063\t2.708\t0.199\t1070\n205064\t2.716\t0.2\t545\n205071\t3.421\t0.246\t38\n205072\t3.123\t0.212\t212\n205318\t2.574\t0.174\t88\n205319\t2.626\t0.158\t25\n208947\t2.969\t0.506\t259\n208948\t3.77\t0.671\t12\n208949\t3.392\t0.45\t229\n208951\t3.952\t0.118\t44\n208952\t3.545\t0.115\t138\n208953\t3.197\t0.0\t6\n208954\t3.176\t0.0\t6\n208962\t3.825\t0.039\t12\n208968\t3.44\t-1.0\t1\n208970\t3.273\t0.166\t30\n208971\t3.245\t0.214\t83\n208972\t2.968\t0.115\t113\n208973\t3.43\t0.067\t4\n208974\t3.1\t0.205\t80\n208975\t3.122\t0.204\t60\n208976\t3.551\t0.081\t6\n208977\t4.668\t0.032\t6\n208978\t4.088\t0.033\t12\n208979\t2.917\t0.271\t11\n209158\t2.495\t0.15\t1844\n209159\t2.721\t0.28\t225\n209160\t2.668\t0.436\t78\n209167\t3.017\t0.046\t5\n209168\t3.149\t0.165\t38\n209414\t2.509\t0.24\t10\n209415\t2.427\t0.146\t6\n213050\t4.433\t0.045\t6\n213054\t4.024\t0.099\t10\n213065\t5.027\t-1.0\t1\n213066\t2.704\t0.455\t23\n213067\t3.276\t0.286\t165\n213068\t3.025\t0.034\t20\n213070\t3.155\t0.284\t48\n213071\t3.03\t0.209\t38\n213072\t3.343\t0.249\t92\n213073\t4.446\t0.01\t2\n213074\t4.01\t0.314\t16\n213075\t3.489\t0.653\t32\n213084\t3.85\t0.333\t22\n213086\t4.355\t0.043\t6\n217145\t3.893\t0.312\t115\n217146\t4.057\t0.272\t72\n217147\t3.309\t0.409\t38\n217148\t3.875\t0.23\t104\n217150\t3.913\t0.338\t122\n217151\t4.078\t0.166\t14\n217152\t3.772\t0.324\t18\n217153\t3.606\t0.456\t25\n217154\t3.192\t0.524\t27\n217156\t3.103\t0.413\t22\n217157\t3.863\t0.489\t46\n217158\t3.761\t0.157\t82\n217159\t3.66\t0.413\t6\n217160\t3.35\t0.322\t22\n217161\t2.816\t0.4\t52\n217162\t2.801\t0.247\t341\n217163\t2.814\t0.353\t94\n217164\t2.999\t0.191\t26\n217165\t3.222\t0.232\t233\n217166\t3.234\t0.277\t811\n217167\t3.066\t0.296\t281\n217168\t3.131\t0.252\t665\n217169\t2.89\t0.413\t19\n217170\t3.801\t0.427\t565\n217171\t3.582\t0.409\t827\n217178\t3.927\t0.242\t20\n217180\t3.819\t0.264\t406\n217182\t3.899\t0.189\t7\n217183\t3.278\t-1.0\t1\n225335\t4.276\t0.49\t265\n225336\t4.479\t0.154\t3\n225338\t4.3\t0.0\t2\n225345\t4.243\t-1.0\t1\n225346\t4.182\t-1.0\t1\n225354\t3.789\t-1.0\t1\n225359\t3.687\t0.153\t5\n225360\t4.965\t0.0\t3\n225361\t4.086\t0.077\t5\n225362\t3.938\t0.124\t17\n225363\t4.209\t0.153\t5\n225542\t4.026\t0.331\t148\n225797\t3.377\t0.162\t3\n225798\t3.864\t0.248\t33\n229432\t4.077\t0.334\t194\n229450\t3.552\t0.0\t2\n229455\t3.61\t0.0\t2\n229458\t3.779\t0.076\t5\n229459\t3.62\t0.105\t15\n229638\t3.726\t0.142\t169\n229893\t2.88\t0.148\t17\n233547\t3.663\t0.041\t4\n233733\t3.661\t0.127\t9\n233734\t3.511\t0.169\t1320\n233735\t3.103\t0.143\t699\n233736\t2.952\t0.191\t544\n233743\t4.137\t0.779\t6\n233744\t3.541\t0.191\t21\n233990\t3.356\t0.17\t18\n233991\t3.128\t0.157\t55\n237830\t3.517\t0.262\t866\n237831\t3.053\t0.238\t780\n237832\t2.942\t0.272\t364\n237839\t3.058\t0.078\t3\n237840\t3.557\t0.208\t14\n238086\t3.309\t0.197\t15\n238087\t3.084\t0.184\t40\n241926\t3.477\t0.124\t197\n241927\t3.016\t0.163\t300\n241928\t2.871\t0.151\t320\n241936\t3.545\t0.074\t6\n242182\t3.213\t0.142\t3\n242183\t3.034\t0.049\t17\n246021\t3.749\t-1.0\t1\n246022\t3.442\t0.199\t803\n246023\t3.027\t0.153\t733\n246024\t2.83\t0.16\t737\n246031\t3.715\t0.509\t10\n246032\t3.489\t0.117\t10\n246278\t3.07\t0.175\t7\n246279\t3.018\t0.113\t65\n254014\t3.621\t0.145\t11\n254027\t3.568\t0.017\t3\n254214\t3.39\t0.158\t2671\n254215\t2.996\t0.161\t998\n254216\t2.805\t0.192\t556\n254223\t3.737\t0.252\t40\n254224\t3.439\t0.148\t13\n254470\t3.227\t0.142\t6\n254471\t3.139\t0.173\t24\n254472\t2.911\t0.206\t20\n258111\t2.215\t0.23\t2\n258310\t3.466\t0.262\t271\n258311\t2.927\t0.187\t1239\n258312\t2.76\t0.166\t1083\n258320\t3.383\t0.05\t12\n258566\t3.457\t0.042\t4\n258567\t2.819\t0.112\t11\n262406\t3.329\t0.213\t453\n262407\t2.926\t0.184\t889\n262408\t2.796\t0.186\t1231\n262416\t3.355\t0.381\t10\n262662\t2.919\t0.044\t2\n262663\t2.999\t0.226\t11\n266502\t3.401\t0.179\t388\n266503\t2.89\t0.17\t992\n266504\t2.734\t0.193\t1159\n266511\t3.732\t-1.0\t1\n266512\t3.445\t0.0\t2\n266758\t2.926\t-1.0\t1\n266759\t2.857\t0.231\t31\n270597\t3.404\t0.076\t3\n270598\t3.222\t0.253\t1595\n270599\t2.866\t0.193\t2042\n270600\t2.707\t0.189\t2073\n270607\t3.718\t-1.0\t1\n270608\t3.41\t0.38\t13\n270854\t2.9\t-1.0\t1\n270855\t2.794\t0.192\t30\n274510\t4.001\t0.0\t2\n274694\t3.388\t0.171\t354\n274695\t2.87\t0.172\t416\n274696\t2.724\t0.185\t439\n274950\t2.925\t0.039\t8\n274951\t2.479\t0.253\t12\n278789\t3.409\t0.069\t2\n278790\t3.158\t0.169\t1145\n278791\t2.849\t0.175\t801\n278792\t2.672\t0.179\t577\n278799\t3.625\t-1.0\t1\n278800\t3.474\t0.092\t7\n279046\t2.934\t0.097\t12\n279047\t2.89\t0.09\t12\n282693\t3.553\t0.0\t4\n282886\t3.443\t0.178\t536\n282887\t2.879\t0.209\t168\n282888\t2.725\t0.147\t129\n282895\t3.631\t0.142\t21\n282896\t3.222\t0.01\t8\n283143\t2.527\t0.136\t9\n286981\t3.507\t0.084\t5\n286982\t3.184\t0.155\t3390\n286983\t2.859\t0.153\t1245\n286984\t2.682\t0.159\t499\n286991\t3.54\t0.155\t13\n286992\t3.365\t0.159\t9\n287238\t3.019\t0.138\t24\n287239\t2.855\t0.076\t16\n287240\t2.979\t0.21\t26\n290891\t3.455\t0.011\t2\n290894\t3.514\t-1.0\t1\n291077\t3.476\t0.16\t11\n291078\t3.284\t0.204\t878\n291079\t2.84\t0.187\t460\n291080\t2.635\t0.193\t132\n291088\t3.528\t0.062\t2\n291334\t2.883\t0.06\t12\n291335\t2.948\t0.137\t14\n294994\t3.999\t-1.0\t1\n295173\t3.165\t0.193\t2\n295174\t3.107\t0.177\t2432\n295175\t2.774\t0.182\t341\n295176\t2.652\t0.204\t104\n295184\t3.109\t0.166\t9\n295430\t2.767\t0.112\t12\n295431\t2.657\t0.278\t12\n299081\t2.935\t0.369\t17\n299085\t3.119\t0.214\t10\n299270\t2.918\t0.218\t1400\n299271\t2.737\t0.199\t305\n299272\t2.492\t0.248\t40\n299279\t2.559\t0.037\t4\n299526\t2.488\t0.241\t28\n299527\t2.804\t0.155\t6\n303178\t2.675\t0.417\t434\n303181\t3.222\t0.078\t41\n303182\t3.654\t-1.0\t1\n303183\t3.21\t0.023\t4\n303186\t3.733\t0.044\t8\n303187\t3.128\t0.192\t8\n303365\t2.968\t0.028\t2\n303366\t2.8\t0.213\t3774\n303367\t2.584\t0.227\t2258\n303368\t2.49\t0.297\t312\n303375\t2.904\t0.202\t124\n303376\t3.053\t0.272\t32\n303622\t2.342\t0.159\t6180\n303623\t2.471\t0.154\t65\n307275\t3.127\t0.383\t1161\n307276\t3.076\t0.401\t28\n307277\t3.191\t0.02\t2\n307278\t3.567\t0.117\t38\n307279\t3.269\t0.094\t25\n307280\t3.228\t0.038\t20\n307281\t3.587\t0.03\t2\n307283\t3.205\t0.081\t30\n307292\t3.713\t0.164\t4\n307462\t2.629\t0.197\t1542\n307463\t2.457\t0.127\t3999\n307464\t2.435\t0.206\t207\n307471\t2.732\t0.251\t11\n307472\t2.825\t0.122\t121\n307718\t2.178\t0.135\t8471\n307719\t2.383\t0.156\t157\n307720\t2.299\t0.121\t3\n307728\t2.409\t0.014\t6\n311372\t3.106\t0.207\t804\n311373\t3.181\t-1.0\t1\n311374\t3.548\t0.146\t32\n311375\t3.12\t0.036\t62\n311379\t2.887\t0.06\t22\n311558\t2.597\t0.187\t2089\n311559\t2.357\t0.157\t1187\n311560\t2.486\t0.163\t95\n311567\t3.048\t0.26\t4\n311568\t3.082\t0.291\t44\n311814\t2.108\t0.152\t4572\n311815\t2.428\t0.139\t86\n311824\t2.543\t0.001\t4\n315469\t3.151\t0.278\t365\n315470\t3.549\t0.122\t63\n315471\t3.157\t0.118\t31\n315475\t3.22\t0.061\t39\n315653\t2.65\t0.215\t20\n315654\t2.552\t0.195\t16080\n315655\t2.382\t0.126\t6084\n315656\t2.483\t0.127\t418\n315663\t2.762\t0.14\t37\n315664\t2.816\t0.196\t162\n315910\t2.215\t0.168\t2203\n315911\t2.436\t0.105\t204\n315920\t2.977\t-1.0\t1\n319566\t3.225\t0.248\t552\n319567\t3.367\t0.293\t115\n319568\t3.169\t0.205\t20\n319569\t3.36\t0.238\t50\n319570\t3.392\t0.286\t42\n319571\t2.929\t0.286\t8\n319580\t2.758\t-1.0\t1\n319749\t2.528\t0.434\t19\n319750\t2.374\t0.187\t4743\n319751\t2.364\t0.145\t6282\n319752\t2.404\t0.169\t399\n319759\t2.743\t0.15\t63\n319760\t2.674\t0.189\t638\n320005\t2.585\t0.095\t3\n320006\t2.331\t0.173\t2612\n320007\t2.357\t0.121\t135\n320015\t2.781\t0.006\t2\n320016\t2.898\t0.124\t6\n323663\t3.245\t0.336\t4562\n323664\t3.526\t0.142\t2\n323665\t3.286\t0.102\t22\n323666\t3.306\t0.022\t2\n323667\t2.881\t0.196\t25\n323846\t2.35\t0.15\t3014\n323847\t2.354\t0.137\t1158\n323848\t2.403\t0.253\t6\n323855\t2.769\t0.169\t45\n323856\t2.739\t0.213\t222\n324102\t2.337\t0.19\t1691\n324103\t2.388\t0.156\t45\n327760\t2.694\t0.474\t53\n327941\t2.577\t-1.0\t1\n327942\t2.266\t0.136\t922\n327943\t2.741\t0.157\t953\n327944\t2.741\t0.235\t37\n327951\t2.904\t0.093\t8\n327952\t2.866\t0.242\t158\n328198\t2.38\t0.168\t94\n328199\t2.875\t0.15\t3\n331857\t3.111\t0.21\t60\n332038\t2.542\t0.227\t19\n332039\t2.984\t0.13\t175\n332040\t2.942\t0.295\t20\n332048\t3.399\t0.21\t18\n332294\t2.174\t0.0\t4\n335954\t3.231\t0.429\t436\n336134\t2.776\t0.248\t252\n336135\t2.833\t0.135\t693\n336136\t2.76\t0.196\t207\n336143\t3.678\t0.123\t15\n336144\t3.263\t0.218\t66\n336390\t2.874\t0.122\t4\n336391\t2.699\t0.147\t7\n340051\t3.126\t0.27\t404\n340229\t2.797\t-1.0\t1\n340230\t2.623\t0.24\t807\n340231\t2.807\t0.169\t491\n340232\t2.771\t0.212\t269\n340239\t3.206\t0.012\t2\n340240\t3.325\t0.293\t98\n340486\t2.701\t0.167\t16\n340487\t3.129\t0.34\t2\n368902\t3.528\t0.229\t2516\n368903\t3.077\t0.237\t337\n368904\t2.966\t0.304\t84\n368911\t3.616\t0.121\t7\n368912\t3.535\t0.376\t64\n369158\t3.14\t0.113\t14\n369159\t3.146\t0.323\t8\n369160\t3.179\t0.006\t2\n376924\t3.747\t0.0\t2\n377094\t3.44\t0.227\t3394\n377095\t2.982\t0.201\t1971\n377096\t2.826\t0.242\t770\n377103\t2.998\t0.426\t10\n377104\t3.462\t0.179\t110\n377350\t3.258\t0.302\t74\n377351\t3.02\t0.219\t96\n377352\t3.011\t0.211\t9\n381190\t3.595\t0.186\t49\n381191\t3.016\t0.171\t33\n381192\t3.071\t0.23\t47\n381200\t3.429\t0.053\t8\n385286\t3.676\t0.046\t15\n385287\t3.005\t0.218\t31\n385288\t3.065\t0.263\t47\n385296\t3.84\t0.251\t6\n389383\t3.107\t0.032\t5\n389384\t2.858\t0.234\t18\n389392\t3.517\t0.038\t3\n393479\t3.128\t0.008\t2\n393488\t3.419\t0.064\t3\n397575\t3.05\t0.029\t3\n397576\t2.942\t0.008\t3\n401671\t3.038\t0.025\t3\n401672\t2.931\t0.016\t3\n1073414\t1.594\t0.077\t2000853\n1073415\t1.527\t0.071\t91806\n1073423\t2.001\t0.161\t74\n1073424\t1.981\t0.067\t42\n1073679\t2.05\t0.093\t4\n1073680\t1.992\t0.039\t7\n1077511\t1.504\t0.087\t791\n1077519\t1.839\t0.106\t1862\n1077520\t1.811\t0.097\t38\n1077776\t1.753\t0.11\t23\n1110287\t2.325\t0.221\t8\n16781317\t1.299\t0.138\t2655\n16781318\t1.157\t0.073\t2154155\n16781319\t1.027\t0.087\t120477\n16781320\t1.0\t0.102\t99234\n16781321\t1.229\t0.768\t39\n16781326\t1.698\t0.206\t492\n16781327\t1.491\t0.16\t461\n16781328\t1.414\t0.148\t260\n16781329\t1.46\t0.267\t16\n16781344\t1.768\t0.207\t170\n16781345\t1.64\t0.233\t13\n16781346\t1.184\t0.24\t11\n16781347\t1.193\t0.14\t10\n16781365\t1.612\t0.001\t2\n16781573\t1.266\t-1.0\t1\n16781574\t1.083\t0.054\t1399983\n16781575\t1.026\t0.08\t10086\n16781576\t1.035\t0.083\t62\n16781583\t1.555\t0.169\t170\n16781830\t1.082\t0.064\t1503\n16781831\t1.012\t0.075\t9\n16797701\t2.055\t0.175\t200\n16797702\t1.917\t0.136\t1968\n16797703\t1.74\t0.145\t4307\n16797704\t1.627\t0.111\t7838\n16797705\t1.579\t0.095\t5410\n16797710\t2.643\t0.184\t32\n16797711\t2.367\t0.191\t736\n16797712\t2.279\t0.266\t207\n16797713\t2.22\t0.188\t513\n16797728\t2.564\t0.128\t21\n16797729\t2.481\t0.101\t5\n16797730\t2.577\t0.148\t48\n16797731\t2.401\t0.231\t244\n16797748\t2.725\t0.14\t4\n16797749\t2.87\t0.374\t77\n16797958\t1.838\t0.106\t11969\n16797959\t1.825\t0.108\t1915\n16797967\t2.268\t0.255\t20\n16798214\t1.868\t0.127\t268\n16798215\t1.772\t0.141\t42\n16798224\t2.259\t-1.0\t1\n16801798\t1.819\t0.111\t650683\n16801799\t1.721\t0.104\t153277\n16801800\t1.674\t0.089\t194202\n16801801\t1.483\t0.083\t37441\n16801806\t2.363\t0.237\t31159\n16801807\t2.243\t0.189\t5173\n16801808\t2.128\t0.132\t11654\n16801809\t2.024\t0.124\t15529\n16801824\t2.527\t0.248\t999\n16801825\t2.422\t0.325\t206\n16801826\t2.378\t0.18\t872\n16801827\t2.279\t0.155\t2704\n16801844\t2.666\t0.308\t172\n16801845\t2.522\t0.166\t717\n16802054\t1.762\t0.092\t352183\n16802055\t1.72\t0.097\t14388\n16802056\t1.688\t0.108\t6\n16802063\t2.207\t0.172\t3307\n16802064\t2.105\t0.131\t3350\n16802310\t1.696\t0.091\t7665\n16802311\t1.656\t0.128\t75\n16802320\t2.082\t0.116\t4919\n16805895\t1.623\t0.108\t3208\n16805896\t1.666\t0.126\t2100\n16805897\t1.415\t0.162\t17\n16805902\t2.205\t0.191\t1967\n16805903\t2.079\t0.152\t1682\n16805904\t1.93\t0.103\t249\n16805905\t1.958\t0.111\t24\n16805920\t2.362\t0.195\t1315\n16805921\t2.241\t0.173\t128\n16805922\t2.3\t0.193\t96\n16805923\t2.162\t0.118\t32\n16805940\t2.674\t0.201\t118\n16805941\t2.473\t0.194\t38\n16806150\t1.59\t0.082\t164495\n16806151\t1.574\t0.081\t7627\n16806159\t1.986\t0.139\t3197\n16806160\t2.009\t0.09\t401\n16806406\t1.512\t0.069\t153\n16806407\t1.394\t0.086\t533\n16806416\t1.9\t0.093\t9234\n16809992\t1.741\t0.264\t647\n16809998\t1.995\t0.155\t7813\n16809999\t1.925\t0.144\t1208\n16810000\t1.916\t0.201\t51\n16810001\t1.673\t0.087\t998\n16810016\t2.152\t0.242\t1021\n16810017\t1.796\t0.244\t670\n16810018\t2.006\t0.2\t174\n16810019\t1.965\t0.381\t4\n16810036\t2.404\t0.302\t460\n16810037\t2.386\t0.22\t205\n16810246\t1.562\t0.08\t163577\n16810247\t1.393\t0.102\t21110\n16810255\t1.861\t0.129\t9068\n16810256\t1.926\t0.104\t123\n16810502\t1.408\t0.07\t41\n16810503\t1.354\t0.073\t7\n16810512\t1.734\t0.114\t5958\n16814089\t1.446\t0.064\t5\n16814094\t1.979\t0.162\t679\n16814095\t1.82\t0.095\t9307\n16814096\t1.773\t0.108\t625\n16814097\t1.615\t0.213\t20\n16814112\t2.054\t0.303\t209\n16814113\t1.845\t0.169\t415\n16814114\t2.218\t0.193\t16\n16814115\t1.836\t0.098\t23\n16814132\t1.837\t0.254\t287\n16814133\t2.14\t0.294\t84\n16814342\t1.478\t0.072\t44091\n16814343\t1.246\t0.169\t3\n16814351\t1.618\t0.194\t150\n16814352\t1.613\t0.103\t16\n16814608\t1.556\t0.127\t44\n16834574\t3.378\t0.364\t2682\n16834575\t2.882\t0.272\t162\n16834576\t2.626\t0.23\t157\n16834577\t2.549\t0.274\t1011\n16834592\t3.473\t0.286\t507\n16834593\t3.314\t0.371\t24\n16834594\t2.897\t0.327\t42\n16834595\t2.87\t0.389\t169\n16834612\t3.665\t0.536\t15\n16834613\t3.09\t0.417\t78\n16834822\t2.235\t0.149\t6897\n16834823\t2.299\t0.166\t545\n16835078\t2.237\t0.123\t1405\n16835079\t2.102\t0.162\t46\n16838671\t2.751\t0.246\t664\n16838672\t2.368\t0.224\t165\n16838673\t2.447\t0.27\t425\n16838688\t3.18\t0.313\t89\n16838689\t2.773\t0.119\t26\n16838690\t2.582\t0.333\t127\n16838691\t2.563\t0.288\t58\n16838708\t3.027\t0.178\t12\n16838709\t2.915\t0.394\t101\n16838918\t2.115\t0.115\t8657\n16838919\t1.855\t0.183\t439\n16838927\t2.715\t0.169\t135\n16838928\t2.658\t0.069\t6\n16839174\t1.936\t0.185\t23\n16839175\t2.101\t0.064\t2\n16839184\t2.948\t-1.0\t1\n16842768\t2.281\t0.201\t1234\n16842769\t2.015\t0.03\t13\n16842784\t2.754\t0.331\t616\n16842785\t2.241\t0.25\t917\n16842786\t2.475\t0.266\t49\n16842787\t2.694\t0.214\t14\n16842804\t3.261\t0.399\t44\n16842805\t3.241\t0.251\t46\n16843014\t2.01\t0.115\t20813\n16843015\t1.834\t0.133\t156\n16843023\t2.464\t0.209\t339\n16843024\t2.494\t0.161\t19\n16843270\t1.935\t0.13\t225\n16843280\t2.489\t0.19\t50\n16846865\t1.992\t0.399\t44\n16846880\t2.745\t0.286\t511\n16846881\t2.728\t0.235\t235\n16846882\t2.384\t0.337\t109\n16846900\t3.03\t0.369\t277\n16846901\t2.906\t0.468\t87\n16847110\t1.969\t0.095\t20992\n16847111\t1.77\t0.177\t16\n16847119\t2.289\t0.187\t1133\n16847120\t2.181\t0.113\t8\n16847366\t1.923\t0.203\t4\n16847376\t2.231\t0.165\t28\n16908320\t3.119\t0.561\t1100\n16908321\t2.481\t0.4\t70\n16908322\t2.624\t0.585\t344\n16908323\t2.911\t0.375\t127\n16908340\t3.181\t0.424\t124\n16908341\t3.271\t0.469\t73\n16908550\t2.342\t0.169\t2233\n16908551\t2.432\t0.175\t414\n16908552\t2.284\t0.062\t14\n16908559\t3.148\t0.384\t31\n16908806\t2.273\t0.193\t233\n16912417\t2.568\t0.286\t226\n16912418\t2.384\t0.277\t52\n16912419\t2.383\t0.526\t37\n16912436\t2.868\t0.102\t3\n16912437\t3.175\t0.376\t43\n16912646\t2.247\t0.155\t563\n16912647\t2.196\t0.137\t13\n16912655\t2.8\t0.063\t13\n16912902\t1.902\t0.429\t4\n16912903\t2.164\t0.088\t7\n16916514\t2.705\t0.288\t257\n16916515\t2.826\t0.329\t129\n16916532\t3.054\t0.385\t17\n16916533\t3.596\t0.24\t25\n16916742\t2.215\t0.127\t1635\n16916743\t1.989\t0.141\t55\n16916751\t2.686\t0.173\t226\n16916752\t2.64\t-1.0\t1\n16916998\t2.197\t0.154\t196\n16920611\t3.015\t0.264\t187\n16920628\t3.279\t0.255\t239\n16920629\t3.195\t0.403\t63\n16920838\t2.164\t0.108\t13670\n16920839\t2.524\t0.134\t43\n16920847\t2.176\t0.207\t50\n16920848\t2.785\t0.214\t9\n16921094\t2.117\t0.097\t16\n16990260\t3.228\t0.382\t40\n16990261\t3.606\t0.35\t107\n16990470\t2.49\t0.154\t812\n16990471\t2.273\t0.267\t50\n16990472\t3.042\t-1.0\t1\n16990479\t3.31\t0.124\t16\n16990726\t2.535\t0.253\t17\n16994357\t3.54\t0.331\t863\n16994566\t2.373\t0.134\t5123\n16994567\t2.597\t0.154\t114\n16994575\t3.258\t0.143\t18\n16994822\t2.308\t0.108\t265\n16994823\t2.683\t0.0\t2\n17846534\t1.807\t0.004\t6\n17850630\t1.686\t0.082\t194786\n17850631\t1.634\t0.082\t26774\n17850632\t1.542\t0.034\t5\n17850639\t2.101\t0.105\t7379\n17850640\t2.077\t0.073\t1123\n17850886\t1.624\t0.087\t18128\n17850887\t1.587\t0.111\t90\n17850896\t2.041\t0.093\t13784\n17854727\t1.605\t0.077\t739\n17854735\t1.988\t0.193\t481\n17854736\t2.01\t0.157\t103\n17854982\t1.518\t0.05\t118\n17854983\t1.299\t0.035\t2\n17854991\t2.079\t0.0\t2\n17854992\t1.872\t0.081\t564\n17858824\t1.212\t0.0\t2\n17887503\t2.816\t0.232\t70\n17887504\t2.775\t-1.0\t1\n17887750\t2.038\t0.159\t42\n17887760\t2.72\t0.033\t2\n17891600\t2.75\t0.187\t12\n17891856\t2.984\t-1.0\t1\n18899462\t1.586\t0.08\t708\n18899472\t2.099\t0.117\t10\n18940432\t2.57\t0.156\t5\n33611790\t2.862\t0.219\t107\n33611809\t3.023\t-1.0\t1\n33611810\t1.697\t0.952\t3\n33611828\t3.382\t-1.0\t1\n33612038\t2.267\t0.06\t19\n33612039\t2.076\t0.084\t9\n33612040\t2.213\t0.116\t7\n33612047\t3.299\t0.356\t32\n33612048\t2.829\t-1.0\t1\n33612294\t2.498\t0.086\t4\n33624117\t2.68\t0.0\t2\n33624326\t2.051\t0.166\t7\n33624328\t1.644\t0.085\t2970\n33685536\t3.587\t0.192\t22\n33685537\t3.362\t0.618\t7\n33685539\t2.768\t-1.0\t1\n33685766\t2.418\t0.096\t23\n33685767\t2.3\t-1.0\t1\n33685768\t1.911\t0.307\t75\n33685775\t3.061\t0.334\t74\n33685776\t3.159\t0.258\t11\n33686022\t2.244\t-1.0\t1\n33689633\t2.268\t0.368\t3\n33689634\t3.2\t0.0\t2\n33689653\t3.142\t0.113\t4\n33689862\t1.874\t0.13\t10\n33689863\t1.999\t-1.0\t1\n33689864\t2.138\t0.163\t58\n33689871\t2.766\t0.099\t11\n33689872\t2.702\t0.904\t6\n33693731\t3.211\t0.049\t4\n33693748\t3.96\t0.381\t8\n33693749\t3.32\t0.198\t7\n33693958\t2.081\t0.047\t6\n33693959\t2.112\t0.21\t9\n33693960\t1.88\t0.205\t162\n33693967\t2.503\t0.176\t332\n33697844\t3.872\t0.0\t2\n33698055\t1.967\t-1.0\t1\n33698056\t1.612\t0.028\t8\n33767476\t2.657\t-1.0\t1\n33767477\t3.446\t-1.0\t1