/*
 * epfl-safety v2.0.1
 * Calculate hazard for a reaction
 * https://github.com/cheminfo/epfl-safety#readme
 *
 * Licensed under the MIT license.
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.EPFLSafety = {}));
})(this, (function (exports) { 'use strict';

    var _documentCurrentScript = typeof document !== 'undefined' ? document.currentScript : null;
    class LogEvent extends CustomEvent {
      constructor(data) {
        super('log', {
          detail: data
        });
      }
    }
    class ChangeEvent extends CustomEvent {
      constructor(data) {
        super('change', {
          detail: data
        });
      }
    }

    var e = class extends EventTarget {
      dispatchTypedEvent(s, t) {
        return super.dispatchEvent(t);
      }
    };

    const levels = {
      values: {
        fatal: 60,
        error: 50,
        warn: 40,
        info: 30,
        debug: 20,
        trace: 10,
        silent: 0
      },
      labels: {
        0: 'silent',
        10: 'trace',
        20: 'debug',
        30: 'info',
        40: 'warn',
        50: 'error',
        60: 'fatal'
      }
    };

    /**
     * A FIFO logger that stores the most recent events in an array.
     */
    class FifoLogger extends e {
      lastID;
      initialOptions;
      events;
      uuids;
      levelAsNumber;
      limit;
      bindings;
      level;
      constructor(options = {}) {
        super();
        this.lastID = {
          id: 0
        };
        this.initialOptions = options;
        this.uuids = [crypto.randomUUID()];
        this.events = [];
        this.level = options.level || 'info';
        this.levelAsNumber = levels.values[this.level];
        this.limit = options.limit ?? 1000;
        this.bindings = options.bindings ?? {};
      }
      setLevel(level) {
        this.level = level;
        this.levelAsNumber = levels.values[level];
      }
      setLimit(limit) {
        this.limit = limit;
        this.checkSize();
      }
      checkSize() {
        if (this.events.length > this.limit) {
          this.events.splice(0, this.events.length - this.limit);
        }
      }
      /**
       * Remove events from the current logger and its children.
       */
      clear() {
        for (let i = this.events.length - 1; i >= 0; i--) {
          if (this.events[i].uuids.includes(this.uuids[0])) {
            this.events.splice(i, 1);
          }
        }
        this.dispatchTypedEvent('change', new ChangeEvent({
          logs: this.events,
          info: {
            depth: this.uuids.length
          }
        }));
      }
      /**
       * Get a filtered list of all log events.
       * @param options - Options.
       * @returns Array of filtered logs.
       */
      getLogs(options = {}) {
        const {
          level,
          minLevel,
          includeChildren
        } = options;
        let logs = this.events.slice();
        if (includeChildren) {
          logs = logs.filter(log => log.uuids.includes(this.uuids[0]));
        } else {
          logs = logs.filter(log => log.uuids[0] === this.uuids[0]);
        }
        if (level) {
          const levelNumber = levels.values[level];
          if (Number.isNaN(levelNumber)) {
            throw new Error('Invalid level');
          }
          logs = logs.filter(log => log.level === levelNumber);
        }
        if (minLevel) {
          const levelNumber = levels.values[minLevel];
          if (Number.isNaN(levelNumber)) {
            throw new Error('Invalid level');
          }
          logs = logs.filter(log => log.level >= levelNumber);
        }
        return logs;
      }
      /**
       * Create a child logger with additional bindings.
       * @param bindings - an object of key-value pairs to include in log lines as properties.
       * @returns The new child logger.
       */
      child(bindings) {
        const newFifoLogger = new FifoLogger(this.initialOptions);
        newFifoLogger.events = this.events;
        newFifoLogger.uuids = [crypto.randomUUID(), ...this.uuids];
        newFifoLogger.lastID = this.lastID;
        newFifoLogger.bindings = {
          ...this.bindings,
          ...bindings
        };
        newFifoLogger.addEventListener('change', event => {
          this.dispatchTypedEvent('change', new ChangeEvent(event.detail));
        });
        return newFifoLogger;
      }
      trace(value, message) {
        this.#addEvent(levels.values.trace, value, message);
      }
      debug(value, message) {
        this.#addEvent(levels.values.debug, value, message);
      }
      info(value, message) {
        this.#addEvent(levels.values.info, value, message);
      }
      warn(value, message) {
        this.#addEvent(levels.values.warn, value, message);
      }
      error(value, message) {
        this.#addEvent(levels.values.error, value, message);
      }
      fatal(value, message) {
        this.#addEvent(levels.values.fatal, value, message);
      }
      #addEvent(level, value, message) {
        if (level < this.levelAsNumber) return;
        const event = {
          id: ++this.lastID.id,
          level,
          levelLabel: levels.labels[level],
          time: Date.now(),
          uuids: this.uuids,
          message: ''
        };
        if (value instanceof Error) {
          event.message = value.toString();
          event.error = value;
          event.meta = {
            ...this.bindings
          };
        } else if (message !== undefined && typeof value === 'object') {
          event.message = message;
          event.meta = {
            ...this.bindings,
            ...value
          };
        } else if (message === undefined && typeof value === 'string') {
          event.message = value;
          event.meta = {
            ...this.bindings
          };
        } else {
          throw new Error('Invalid arguments');
        }
        this.events.push(event);
        this.checkSize();
        this.dispatchTypedEvent('log', new LogEvent({
          log: event,
          logs: this.events,
          info: {
            depth: this.uuids.length
          }
        }));
        this.dispatchTypedEvent('change', new ChangeEvent({
          logs: this.events,
          info: {
            depth: this.uuids.length
          }
        }));
      }
    }

    const redAcuteToxicitiesHazards = new Set(['H300', 'H310', 'H330']);
    const rules$3 = [{
      description: 'Extreme acute toxicity',
      info: {
        highOrExtreme: true
      },
      matcher: {
        hazards: ['H300', 'H310', 'H330']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 6
      }, {
        lessThan: +Infinity,
        score: 12
      }]
    }, {
      description: 'High acute toxicity',
      info: {
        highOrExtreme: true
      },
      matcher: {
        hazards: ['H311', 'H318', 'H331']
      },
      masses: [{
        lessThan: 0.001,
        score: 2
      }, {
        lessThan: 0.003,
        score: 3
      }, {
        lessThan: +Infinity,
        score: 4
      }]
    }, {
      description: 'Medium acute toxicity',
      info: {
        highOrExtreme: false
      },
      matcher: {
        hazards: ['H302', 'H303', 'H305', 'H312', 'H313', 'H330', 'H332', 'H333', 'H334']
      },
      masses: [{
        lessThan: 0.001,
        score: 1
      }, {
        lessThan: 0.003,
        score: 2
      }, {
        lessThan: +Infinity,
        score: 3
      }]
    }, {
      description: 'Low acute toxicity',
      info: {
        highOrExtreme: false
      },
      matcher: {
        hazards: ['H301', 'H304', 'H315', 'H316', 'H317', 'H319,H320', 'H334', 'H335', 'H336']
      },
      masses: [{
        lessThan: 0.001,
        score: 0.5
      }, {
        lessThan: 0.003,
        score: 1
      }, {
        lessThan: +Infinity,
        score: 2
      }]
    }, {
      description: 'No known acute toxicity',
      info: {
        highOrExtreme: false
      },
      matcher: {},
      masses: [{
        lessThan: +Infinity,
        score: 0
      }]
    }];
    function appendAcuteToxicity(reagents, report, options) {
      const {
        logger
      } = options;
      let redAcuteToxicities = new Set();
      // what is the worse hazard ?
      for (const rule of rules$3) {
        for (const reagent of reagents) {
          if (reagent.safety.acuteToxicity) continue;
          appendRed$1(reagent, redAcuteToxicities);
          const matching = isMatching$3(reagent, rule);
          if (!matching) continue;
          reagent.safety.acuteToxicity = {
            description: rule.description,
            matchingHazards: matching.matchingHazards || [],
            info: rule.info,
            score: rule.masses.find(mass => reagent.mass.value <= mass.lessThan).score
          };
        }
      }

      /*
      If high or extreme are present only take sum of high and extreme values.
      If no high or extreme, simply take the score for the one reagent that leads to the highest score.
      */
      const highOrExtremes = reagents.filter(reagent => reagent.safety.acuteToxicity.info.highOrExtreme);
      if (highOrExtremes.length === 0) {
        logger?.info('AcuteToxicity: No high or extreme acute toxicity. Taking the highest score of the other.');
        report.scores.acuteToxicity = Math.max(...reagents.map(reagent => reagent.safety.acuteToxicity.score));
      } else {
        logger?.info('AcuteToxicity: High or extreme acute toxicity. Taking the sum of them.');
        report.scores.acuteToxicity = highOrExtremes.reduce((sum, reagent) => sum + reagent.safety.acuteToxicity.score, 0);
      }
      report.redAcuteToxicity = [...redAcuteToxicities];
    }
    function isMatching$3(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }
    function appendRed$1(reagent, redAcuteToxicities) {
      for (const hazard of reagent.hazards) {
        if (redAcuteToxicitiesHazards.has(hazard)) {
          redAcuteToxicities.add(hazard);
        }
      }
    }

    const rules$2 = [{
      description: 'Extreme long-term toxicity',
      matcher: {
        hazards: ['H340', 'H350', 'H360', 'H370']
      },
      volumes: [{
        lessThan: 0.05,
        score: 2
      }, {
        lessThan: 0.5,
        score: 3
      }, {
        lessThan: 1,
        score: 6
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'High long-term toxicity',
      matcher: {
        hazards: ['H341', 'H351', 'H361', 'H371', 'H372', 'H373']
      },
      volumes: [{
        lessThan: 0.05,
        score: 1
      }, {
        lessThan: 0.5,
        score: 2
      }, {
        lessThan: 1,
        score: 4
      }, {
        lessThan: +Infinity,
        score: 8
      }]
    }, {
      description: 'No known long-term toxicity',
      matcher: {},
      volumes: [{
        lessThan: +Infinity,
        score: 0
      }]
    }];
    function appendLongTermToxicity(reagents, report, options = {}) {
      const {
        logger
      } = options;
      // what is the worse hazard ?
      for (const rule of rules$2) {
        for (const reagent of reagents) {
          if (reagent.safety.longTermToxicity) continue;
          const matching = isMatching$2(reagent, rule);
          if (!matching) continue;
          reagent.safety.longTermToxicity = {
            description: rule.description,
            matchingHazards: matching.matchingHazards || [],
            score: rule.volumes.find(volume => reagent.volume.value <= volume.lessThan).score
          };
        }
      }
      logger?.info('LongTermToxicity: Taking the highest score.');
      report.scores.longTermToxicity = Math.max(...reagents.map(reagent => reagent.safety.longTermToxicity.score));
    }
    function isMatching$2(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }

    function appendMiscellaneous(reagents, report, options = {}) {
      const {
        miscellaneous = {}
      } = options;
      const {
        highVoltage,
        dustRisk,
        reactionTime,
        temperature,
        gasLines,
        pressure,
        overheadStirring
      } = miscellaneous;
      const results = [];
      results.push(getDustRisk(dustRisk, options));
      results.push(getReactionTime(reactionTime, options));
      results.push(getTemperature(temperature, options));
      results.push(getGasLines(gasLines, options));
      results.push(getPressure(pressure, options));
      results.push(getOverheadStirring(overheadStirring, options));
      results.push(getHighVoltage(highVoltage, options));
      report.miscellaneous = results;
    }
    function getDustRisk(dustRisk, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Dust risk',
        info: '(for more than 5g of solids)',
        scores: []
      };
      switch (dustRisk) {
        case 'yes':
          entry.scores.push({
            remarks: 'Mask still required for all quantities of fine particles!',
            label: 'Fine powders, fibers air suspended',
            score: 4
          });
          break;
        case 'none':
          entry.scores.push({
            label: 'None',
            score: 0
          });
          break;
        default:
          logger.error(`Unknown dustRisk: ${dustRisk}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getReactionTime(reactionTime, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Reaction time',
        info: '',
        scores: []
      };
      switch (reactionTime) {
        case 'daytime':
          entry.scores.push({
            label: 'Daytime',
            score: 0
          });
          break;
        case 'overnight':
          entry.scores.push({
            label: 'Overnight',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown reactionTime: ${reactionTime}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getTemperature(temperature, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Temperature',
        info: '',
        scores: []
      };
      switch (temperature) {
        case '<100':
          entry.scores.push({
            label: '< 100°C',
            score: 0
          });
          break;
        case '100-200':
          entry.scores.push({
            label: '100-200°C',
            score: 1
          });
          break;
        case 'extreme':
          entry.scores.push({
            label: '> 200°C or < -30°C',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown temperature: ${temperature}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getGasLines(gasLines, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Handling High Pressure Gas Lines',
        info: '(Gas line pressure)',
        scores: []
      };
      const inertDescription = 'Inert (e.g., N2, He, Ar, CO2)';
      switch (gasLines.inert) {
        case 'none':
          entry.scores.push({
            description: inertDescription,
            label: 'None',
            score: 0
          });
          break;
        case '<20':
          entry.scores.push({
            description: inertDescription,
            label: '<20 barg',
            score: 0
          });
          break;
        case '20-40':
          entry.scores.push({
            description: inertDescription,
            label: '20-40 barg',
            score: 1
          });
          break;
        case '>40':
          entry.scores.push({
            description: inertDescription,
            label: '>40 barg',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown gasLines.inert: ${gasLines.inert}`);
      }
      const reactiveDescription = 'Reactive/Hazardous (e.g., CO, NH3, H2, hydrocarbons)';
      switch (gasLines.reactive) {
        case 'none':
          entry.scores.push({
            description: reactiveDescription,
            label: 'None',
            score: 0
          });
          break;
        case '<20':
          entry.scores.push({
            description: reactiveDescription,
            label: '<20 barg',
            score: 2
          });
          break;
        case '20-40':
          entry.scores.push({
            description: reactiveDescription,
            label: '20-40 barg',
            score: 3
          });
          break;
        case '>40':
          entry.scores.push({
            description: reactiveDescription,
            label: '>40 barg',
            score: 4
          });
          break;
        default:
          logger.error(`Unknown gasLines.inert: ${gasLines.reactive}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getPressure(pressure, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'For pressurized reactions in Parr/flow',
        info: 'Did you pass the PARR / Flow reactor exam and sign the pressurized SOP and PHA?',
        scores: []
      };
      const parrLowDescription = `Parr < 50 mL or flow reactor (1/4'')`;
      switch (pressure.parrLow) {
        case 'none':
          entry.scores.push({
            description: parrLowDescription,
            label: 'None',
            score: 0
          });
          break;
        case '1-20':
          entry.scores.push({
            description: parrLowDescription,
            label: '1-20 barg',
            score: 1
          });
          break;
        case '20-100':
          entry.scores.push({
            description: parrLowDescription,
            label: '20-100 barg',
            score: 2
          });
          break;
        case '100-120':
          entry.scores.push({
            description: parrLowDescription,
            label: '100-120 barg',
            score: 12
          });
          break;
        case '>120':
          entry.scores.push({
            description: parrLowDescription,
            label: '>120 barg',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.parrLow: ${pressure.parrLow}`);
      }
      const parrHighDescription = `Parr > 50 mL or flow reactor (1/2'')`;
      switch (pressure.parrHigh) {
        case 'none':
          entry.scores.push({
            description: parrHighDescription,
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            description: parrHighDescription,
            label: 'Yes',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.parrHigh: ${pressure.parrHigh}`);
      }
      const pRatedDescription = 'Glass (P-Rated)';
      switch (pressure.pRated) {
        case 'none':
          entry.scores.push({
            description: pRatedDescription,
            label: 'None',
            score: 0
          });
          break;
        case '1-3':
          entry.scores.push({
            description: pRatedDescription,
            label: '1-3 barg',
            score: 1
          });
          break;
        case '3-5':
          entry.scores.push({
            description: pRatedDescription,
            label: '3-5 barg',
            score: 12
          });
          break;
        case '5-8':
          entry.scores.push({
            description: pRatedDescription,
            label: '5-8 barg',
            score: 18
          });
          break;
        case '>8':
          entry.scores.push({
            description: pRatedDescription,
            label: '>8 barg',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.pRated: ${pressure.pRated}`);
      }
      const nonRatedDescription = 'Glass (Not rated)';
      switch (pressure.nonRated) {
        case 'none':
          entry.scores.push({
            description: nonRatedDescription,
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            description: nonRatedDescription,
            label: 'Yes',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown pressure.nonRated: ${pressure.nonRated}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getOverheadStirring(overheadStirring, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'Overheads stirring ?',
        info: '',
        scores: []
      };
      switch (overheadStirring) {
        case 'none':
          entry.scores.push({
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            label: 'Yes',
            score: 2
          });
          break;
        default:
          logger.error(`Unknown overheadStirring: ${overheadStirring}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }
    function getHighVoltage(highVoltage, options) {
      const {
        logger
      } = options;
      const entry = {
        label: 'High voltage ?',
        info: '',
        scores: []
      };
      switch (highVoltage) {
        case 'none':
          entry.scores.push({
            label: 'None',
            score: 0
          });
          break;
        case 'yes':
          entry.scores.push({
            label: 'Yes',
            score: 100
          });
          break;
        default:
          logger.error(`Unknown highVoltage: ${highVoltage}`);
      }
      entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
      return entry;
    }

    const redReactivityAndInstabilityHazards = new Set(['H260', 'H261', 'H250', 'H271', 'H272', 'H200', 'H201', 'H202', 'H203', 'H240', 'H241', 'H242', 'H251', 'H252']);
    const rules$1 = [{
      description: 'Flammable gas generation',
      matcher: {
        hazards: ['H260', 'H261']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 5
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Pyrophoric',
      matcher: {
        hazards: ['H250']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 6
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Oxidizer',
      matcher: {
        hazards: ['H270', 'H271', 'H272']
      },
      masses: [{
        lessThan: 0.001,
        score: 1
      }, {
        lessThan: 0.003,
        score: 3
      }, {
        lessThan: +Infinity,
        score: 6
      }]
    }, {
      description: 'Explosive / Self-heating',
      matcher: {
        hazards: ['H200', 'H201', 'H202', 'H203', 'H240', 'H241', 'H242', 'H251', 'H252']
      },
      masses: [{
        lessThan: 0.001,
        score: 3
      }, {
        lessThan: 0.003,
        score: 5
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Corrosive',
      matcher: {
        hazards: ['H314']
      },
      masses: [{
        lessThan: 0.001,
        score: 0.5
      }, {
        lessThan: 0.003,
        score: 1
      }, {
        lessThan: +Infinity,
        score: 2
      }]
    }, {
      description: 'No reactivity or instability hazards known',
      matcher: {},
      masses: [{
        lessThan: +Infinity,
        score: 0
      }]
    }];
    function appendReactivityAndInstability(reagents, report) {
      // what is the worse hazard ?
      let sum = 0;
      let redReactivityAndInstabilities = new Set();
      for (const rule of rules$1) {
        for (const reagent of reagents) {
          if (!reagent.safety.reactivityAndInstabilities) {
            reagent.safety.reactivityAndInstabilities = [];
          }
          const matching = isMatching$1(reagent, rule);
          appendRed(reagent, redReactivityAndInstabilities);
          if (!matching) continue;
          const score = rule.masses.find(mass => reagent.mass.value <= mass.lessThan).score;
          if (score > 0 || reagent.safety.reactivityAndInstabilities.length === 0) {
            reagent.safety.reactivityAndInstabilities.push({
              description: rule.description,
              matchingHazards: matching.matchingHazards || [],
              score
            });
            sum += score;
          }
        }
      }
      report.redReactivityAndInstabilities = [...redReactivityAndInstabilities];
      report.scores.reactivityAndInstability = sum;
    }
    function appendRed(reagent, redReactivity) {
      for (const hazard of reagent.hazards) {
        if (redReactivityAndInstabilityHazards.has(hazard)) {
          redReactivity.add(hazard);
        }
      }
    }
    function isMatching$1(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }

    const rules = [{
      description: 'Very flammable, low bp',
      matcher: {
        hazards: ['H222', 'H224', 'H225'],
        bpLessThan: 35
      },
      volumes: [{
        lessThan: 0.01,
        score: 1.5
      }, {
        lessThan: 0.05,
        score: 3
      }, {
        lessThan: 0.5,
        score: 4
      }, {
        lessThan: 1,
        score: 5
      }, {
        lessThan: +Infinity,
        score: 10
      }]
    }, {
      description: 'Very flammable, high bp',
      matcher: {
        hazards: ['H222', 'H224', 'H225']
      },
      volumes: [{
        lessThan: 0.01,
        score: 1
      }, {
        lessThan: 0.05,
        score: 2
      }, {
        lessThan: 0.5,
        score: 3
      }, {
        lessThan: 1,
        score: 4
      }, {
        lessThan: +Infinity,
        score: 8
      }]
    }, {
      description: 'Flammable, low bp',
      matcher: {
        hazards: ['H223', 'H226', 'H227', 'H228'],
        bpLessThan: 35
      },
      volumes: [{
        lessThan: 0.01,
        score: 0.5
      }, {
        lessThan: 0.05,
        score: 1
      }, {
        lessThan: 0.5,
        score: 2
      }, {
        lessThan: 1,
        score: 3
      }, {
        lessThan: +Infinity,
        score: 6
      }]
    }, {
      description: 'Flammable, high bp',
      matcher: {
        hazards: ['H223', 'H226', 'H227', 'H228']
      },
      volumes: [{
        lessThan: 0.01,
        score: 0
      }, {
        lessThan: 0.05,
        score: 0.5
      }, {
        lessThan: 0.5,
        score: 1
      }, {
        lessThan: 1,
        score: 2
      }, {
        lessThan: +Infinity,
        score: 4
      }]
    }, {
      description: 'Not flammable',
      matcher: {},
      volumes: [{
        lessThan: 0.5,
        score: 0
      }, {
        lessThan: 1,
        score: 1
      }, {
        lessThan: +Infinity,
        score: 3
      }]
    }];
    function appendVolumeAndFlammability(reagents, report, options) {
      const totalVolume = getTotalVolume(reagents, options);
      // what is the worse hazard ?
      let bestScore = 0;
      for (const rule of rules) {
        for (const reagent of reagents) {
          if (reagent.safety.volumeAndFlammability) continue;
          const matching = isMatching(reagent, rule);
          if (!matching) continue;
          const score = rule.volumes.find(volume => totalVolume <= volume.lessThan).score;
          if (score > bestScore) {
            bestScore = score;
          }
          reagent.safety.volumeAndFlammability = {
            description: rule.description,
            matchingHazards: matching.matchingHazards || [],
            score
          };
        }
      }
      report.scores.volumeAndFlammability = bestScore;
    }
    function isMatching(reagent, rule) {
      if (!rule.matcher.hazards) return true;
      if (!rule.matcher.bpLessThan && reagent.bp > rule.matcher.bpLessThan) {
        return false;
      }
      const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
      if (matchingHazards.length === 0) return false;
      return {
        matchingHazards
      };
    }
    function getTotalVolume(reagents, options) {
      const {
        logger
      } = options;
      let totalVolume = 0;
      for (const reagent of reagents) {
        if (reagent.volume) {
          totalVolume += reagent.volume.value;
        } else {
          logger.warn(`reagent ${reagent.smiles} has no volume`);
          totalVolume += reagent.mass.value;
        }
      }
      return totalVolume;
    }

    function filterReagents(reagents, options) {
      const reagentsKind = options.miscellaneous?.reagentsKind;
      if (reagentsKind === 'nonWorkup') {
        reagents = reagents.filter(reagent => reagent.kind?.replaceAll(/[^a-z]/gi, '') !== 'workup');
      } else if (reagentsKind === 'onlyWorkup') {
        reagents = reagents.filter(reagent => reagent.kind?.replaceAll(/[^a-z]/gi, '') === 'workup');
      }
      return reagents;
    }

    // Maximum safe timeout value for setTimeout in JavaScript (2^31 - 1 milliseconds)
    // This is approximately 24.8 days
    const MAX_TIMEOUT = 2_147_483_647;

    // Brand symbol to identify our timeout/interval objects
    // Use Symbol.for to ensure cross-copy compatibility (monorepos, hoisted deps, etc.)
    const brandSymbol = Symbol.for('sindresorhus/unlimited-timeout#brand');
    function setTimeout$1(callback, delay, ...arguments_) {
      if (typeof callback !== 'function') {
        throw new TypeError('Expected callback to be a function');
      }

      // Coerce delay to number, matching native setTimeout behavior
      delay ??= 0;
      delay = Number(delay);
      let shouldUnref = false;
      const timeout = {
        [brandSymbol]: true,
        id: undefined,
        cleared: false,
        ref() {
          shouldUnref = false;
          timeout.id?.ref?.();
          return timeout;
        },
        unref() {
          shouldUnref = true;
          timeout.id?.unref?.();
          return timeout;
        }
      };

      // Treat delays beyond MAX_SAFE_INTEGER as Infinity (precision loss)
      // and positive Infinity means wait forever (never fire)
      if (delay === Number.POSITIVE_INFINITY || delay > Number.MAX_SAFE_INTEGER) {
        return timeout;
      }

      // Clamp invalid values to 0 (NaN, negative numbers result in immediate firing)
      if (!Number.isFinite(delay) || delay < 0) {
        delay = 0;
      }

      // Track target timestamp to avoid overshoot when chunks fire late
      const targetTime = performance.now() + delay;
      const schedule = remainingDelay => {
        if (timeout.cleared) {
          return;
        }
        if (remainingDelay <= MAX_TIMEOUT) {
          // Final timeout - execute callback
          timeout.id = globalThis.setTimeout(() => {
            if (!timeout.cleared) {
              callback(...arguments_);
            }
          }, remainingDelay);
          if (shouldUnref) {
            timeout.id?.unref?.();
          }
        } else {
          // Schedule next chunk
          timeout.id = globalThis.setTimeout(() => {
            const now = performance.now();
            const remaining = Math.max(0, targetTime - now);
            schedule(remaining);
          }, MAX_TIMEOUT);
          if (shouldUnref) {
            timeout.id?.unref?.();
          }
        }
      };
      schedule(delay);
      return timeout;
    }
    function clearTimeout$1(timeout) {
      if (!timeout || typeof timeout !== 'object' || !timeout[brandSymbol]) {
        return;
      }
      timeout.cleared = true;
      if (timeout.id !== undefined) {
        globalThis.clearTimeout(timeout.id);
        timeout.id = undefined;
      }
    }

    const clearMethods = new WeakMap();
    function createDelay({
      clearTimeout: defaultClear,
      setTimeout: defaultSet
    } = {}) {
      // We cannot use `async` here as we need the promise identity.
      return (milliseconds, {
        value,
        signal
      } = {}) => {
        if (signal?.aborted) {
          return Promise.reject(signal.reason);
        }
        let timeoutId;
        let settle;
        let rejectFunction;
        const clear = defaultClear ?? clearTimeout;
        const signalListener = () => {
          clear(timeoutId);
          rejectFunction(signal.reason);
        };
        const cleanup = () => {
          if (signal) {
            signal.removeEventListener('abort', signalListener);
          }
        };
        const delayPromise = new Promise((resolve, reject) => {
          settle = () => {
            cleanup();
            resolve(value);
          };
          rejectFunction = reject;
          timeoutId = (defaultSet ?? setTimeout)(settle, milliseconds);
        });
        if (signal) {
          signal.addEventListener('abort', signalListener, {
            once: true
          });
        }
        clearMethods.set(delayPromise, () => {
          clear(timeoutId);
          timeoutId = null;
          settle();
        });
        return delayPromise;
      };
    }
    const delay = createDelay({
      setTimeout: setTimeout$1,
      clearTimeout: clearTimeout$1
    });

    const ghs = {
      "hStatements": [{
        "code": "H200",
        "statement": "(Deleted) Unstable Explosive"
      }, {
        "code": "H201",
        "statement": "(Deleted) Explosive; mass explosion hazard"
      }, {
        "code": "H202",
        "statement": "(Deleted) Explosive; severe projection hazard"
      }, {
        "code": "H203",
        "statement": "(Deleted) Explosive; fire, blast or projection hazard"
      }, {
        "code": "H204",
        "statement": "Fire or projection hazard"
      }, {
        "code": "H205",
        "statement": "(Deleted) May mass explode in fire"
      }, {
        "code": "H206",
        "statement": "Fire, blast or projection hazard; increased risk of explosion if desensitizing agent is reduced"
      }, {
        "code": "H207",
        "statement": "Fire or projection hazard; increased risk of explosion if desensitizing agent is reduced"
      }, {
        "code": "H208",
        "statement": "Fire hazard; increased risk of explosion if desensitizing agent is reduced"
      }, {
        "code": "H209",
        "statement": "Explosive"
      }, {
        "code": "H210",
        "statement": "Very sensitive"
      }, {
        "code": "H211",
        "statement": "Maybe sensitive"
      }, {
        "code": "H220",
        "statement": "Extremely flammable gas"
      }, {
        "code": "H221",
        "statement": "Flammable gas"
      }, {
        "code": "H222",
        "statement": "Extremely flammable aerosol"
      }, {
        "code": "H223",
        "statement": "Flammable aerosol"
      }, {
        "code": "H224",
        "statement": "Extremely flammable liquid and vapor"
      }, {
        "code": "H225",
        "statement": "Highly Flammable liquid and vapor"
      }, {
        "code": "H226",
        "statement": "Flammable liquid and vapor"
      }, {
        "code": "H227",
        "statement": "Combustible liquid"
      }, {
        "code": "H228",
        "statement": "Flammable solid"
      }, {
        "code": "H229",
        "statement": "Pressurized container: may burst if heated"
      }, {
        "code": "H230",
        "statement": "May react explosively even in the absence of air"
      }, {
        "code": "H231",
        "statement": "May react explosively even in the absence of air at elevated pressure and/or temperature"
      }, {
        "code": "H232",
        "statement": "May ignite spontaneously if exposed to air"
      }, {
        "code": "H240",
        "statement": "Heating may cause an explosion"
      }, {
        "code": "H241",
        "statement": "Heating may cause a fire or explosion"
      }, {
        "code": "H242",
        "statement": "Heating may cause a fire"
      }, {
        "code": "H250",
        "statement": "Catches fire spontaneously if exposed to air"
      }, {
        "code": "H251",
        "statement": "Self-heating; may catch fire"
      }, {
        "code": "H252",
        "statement": "Self-heating in large quantities; may catch fire"
      }, {
        "code": "H260",
        "statement": "In contact with water releases flammable gases which may ignite spontaneously"
      }, {
        "code": "H261",
        "statement": "In contact with water releases flammable gas"
      }, {
        "code": "H270",
        "statement": "May cause or intensify fire; oxidizer"
      }, {
        "code": "H271",
        "statement": "May cause fire or explosion; strong Oxidizer"
      }, {
        "code": "H272",
        "statement": "May intensify fire; oxidizer"
      }, {
        "code": "H280",
        "statement": "Contains gas under pressure; may explode if heated"
      }, {
        "code": "H281",
        "statement": "Contains refrigerated gas; may cause cryogenic burns or injury"
      }, {
        "code": "H282",
        "statement": "Extremely flammable chemical under pressure: may explode if heated"
      }, {
        "code": "H283",
        "statement": "Flammable chemical under pressure: may explode if heated"
      }, {
        "code": "H284",
        "statement": "Chemical under pressure: may explode if heated"
      }, {
        "code": "H290",
        "statement": "May be corrosive to metals"
      }, {
        "code": "H300",
        "statement": "Fatal if swallowed"
      }, {
        "code": "H301",
        "statement": "Toxic if swallowed"
      }, {
        "code": "H302",
        "statement": "Harmful if swallowed"
      }, {
        "code": "H303",
        "statement": "May be harmful if swallowed"
      }, {
        "code": "H304",
        "statement": "May be fatal if swallowed and enters airways"
      }, {
        "code": "H305",
        "statement": "May be harmful if swallowed and enters airways"
      }, {
        "code": "H310",
        "statement": "Fatal in contact with skin"
      }, {
        "code": "H311",
        "statement": "Toxic in contact with skin"
      }, {
        "code": "H312",
        "statement": "Harmful in contact with skin"
      }, {
        "code": "H313",
        "statement": "May be harmful in contact with skin"
      }, {
        "code": "H314",
        "statement": "Causes severe skin burns and eye damage"
      }, {
        "code": "H315",
        "statement": "Causes skin irritation"
      }, {
        "code": "H316",
        "statement": "Causes mild skin irritation"
      }, {
        "code": "H317",
        "statement": "May cause an allergic skin reaction"
      }, {
        "code": "H318",
        "statement": "Causes serious eye damage"
      }, {
        "code": "H319",
        "statement": "Causes serious eye irritation"
      }, {
        "code": "H320",
        "statement": "Causes eye irritation"
      }, {
        "code": "H330",
        "statement": "Fatal if inhaled"
      }, {
        "code": "H331",
        "statement": "Toxic if inhaled"
      }, {
        "code": "H332",
        "statement": "Harmful if inhaled"
      }, {
        "code": "H333",
        "statement": "May be harmful if inhaled"
      }, {
        "code": "H334",
        "statement": "May cause allergy or asthma symptoms or breathing difficulties if inhaled"
      }, {
        "code": "H335",
        "statement": "May cause respiratory irritation"
      }, {
        "code": "H336",
        "statement": "May cause drowsiness or dizziness"
      }, {
        "code": "H340",
        "statement": "May cause genetic defects"
      }, {
        "code": "H341",
        "statement": "Suspected of causing genetic defects"
      }, {
        "code": "H350",
        "statement": "May cause cancer"
      }, {
        "code": "H350i",
        "statement": "May cause cancer by inhalation"
      }, {
        "code": "H351",
        "statement": "Suspected of causing cancer"
      }, {
        "code": "H360",
        "statement": "May damage fertility or the unborn child"
      }, {
        "code": "H360F",
        "statement": "May damage fertility"
      }, {
        "code": "H360D",
        "statement": "May damage the unborn child"
      }, {
        "code": "H360FD",
        "statement": "May damage fertility; May damage the unborn child"
      }, {
        "code": "H360Fd",
        "statement": "May damage fertility; Suspected of damaging the unborn child"
      }, {
        "code": "H360Df",
        "statement": "May damage the unborn child; Suspected of damaging fertility"
      }, {
        "code": "H361",
        "statement": "Suspected of damaging fertility or the unborn child"
      }, {
        "code": "H361f",
        "statement": "Suspected of damaging fertility"
      }, {
        "code": "H361d",
        "statement": "Suspected of damaging the unborn child"
      }, {
        "code": "H361fd",
        "statement": "Suspected of damaging fertility; Suspected of damaging the unborn child"
      }, {
        "code": "H362",
        "statement": "May cause harm to breast-fed children"
      }, {
        "code": "H370",
        "statement": "Causes damage to organs"
      }, {
        "code": "H371",
        "statement": "May cause damage to organs"
      }, {
        "code": "H372",
        "statement": "Causes damage to organs through prolonged or repeated exposure"
      }, {
        "code": "H373",
        "statement": "May causes damage to organs through prolonged or repeated exposure"
      }, {
        "code": "H400",
        "statement": "Very toxic to aquatic life"
      }, {
        "code": "H401",
        "statement": "Toxic to aquatic life"
      }, {
        "code": "H402",
        "statement": "Harmful to aquatic life"
      }, {
        "code": "H410",
        "statement": "Very toxic to aquatic life with long lasting effects"
      }, {
        "code": "H411",
        "statement": "Toxic to aquatic life with long lasting effects"
      }, {
        "code": "H412",
        "statement": "Harmful to aquatic life with long lasting effects"
      }, {
        "code": "H413",
        "statement": "May cause long lasting harmful effects to aquatic life"
      }, {
        "code": "H420",
        "statement": "Harms public health and the environment by destroying ozone in the upper atmosphere"
      }, {
        "code": "H300+H310",
        "statement": "Fatal if swallowed or in contact with skin"
      }, {
        "code": "H300+H330",
        "statement": "Fatal if swallowed or if inhaled"
      }, {
        "code": "H310+H330",
        "statement": "Fatal in contact with skin or if inhaled"
      }, {
        "code": "H300+H310+H330",
        "statement": "Fatal if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H301+H311",
        "statement": "Toxic if swallowed or in contact with skin"
      }, {
        "code": "H301+H331",
        "statement": "Toxic if swallowed or if inhaled"
      }, {
        "code": "H311+H331",
        "statement": "Toxic in contact with skin or if inhaled."
      }, {
        "code": "H301+H311+H331",
        "statement": "Toxic if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H302+H312",
        "statement": "Harmful if swallowed or in contact with skin"
      }, {
        "code": "H302+H332",
        "statement": "Harmful if swallowed or if inhaled"
      }, {
        "code": "H312+H332",
        "statement": "Harmful in contact with skin or if inhaled"
      }, {
        "code": "H302+H312+H332",
        "statement": "Harmful if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H303+H313",
        "statement": "May be harmful if swallowed or in contact with skin"
      }, {
        "code": "H303+H333",
        "statement": "May be harmful if swallowed or if inhaled"
      }, {
        "code": "H313+H333",
        "statement": "May be harmful in contact with skin or if inhaled"
      }, {
        "code": "H303+H313+H333",
        "statement": "May be harmful if swallowed, in contact with skin or if inhaled"
      }, {
        "code": "H315+H320",
        "statement": "Cause skin and eye irritation"
      }, {
        "code": "EUH001",
        "statement": "Explosive when dry"
      }, {
        "code": "EUH006",
        "statement": "Explosive with or without contact with air"
      }, {
        "code": "EUH014",
        "statement": "Reacts violently with water"
      }, {
        "code": "EUH018",
        "statement": "In use may form flammable/explosive vapour-air mixture"
      }, {
        "code": "EUH019",
        "statement": "May form explosive peroxides"
      }, {
        "code": "EUH044",
        "statement": "Risk of explosion if heated under confinement"
      }, {
        "code": "EUH029",
        "statement": "Contact with water liberates toxic gas"
      }, {
        "code": "EUH031",
        "statement": "Contact with acids liberates toxic gas"
      }, {
        "code": "EUH032",
        "statement": "Contact with acids liberates very toxic gas"
      }, {
        "code": "EUH066",
        "statement": "Repeated exposure may cause skin dryness or cracking"
      }, {
        "code": "EUH070",
        "statement": "Toxic by eye contact"
      }, {
        "code": "EUH071",
        "statement": "Corrosive to the respiratory tract"
      }, {
        "code": "EUH059",
        "statement": "Hazardous to the ozone layer, superseded by GHS Class 5.1 in the second adaptation to technical progress of CLP."
      }, {
        "code": "EUH201",
        "statement": "Contains lead. Should not be used on surfaces liable to be chewed or sucked by children."
      }, {
        "code": "EUH201A",
        "statement": "Warning! Contains lead."
      }, {
        "code": "EUH202",
        "statement": "Cyanoacrylate. Danger. Bonds skin and eyes in seconds. Keep out of the reach of children."
      }, {
        "code": "EUH203",
        "statement": "Contains chromium(VI). May produce an allergic reaction."
      }, {
        "code": "EUH204",
        "statement": "Contains isocyanates. May produce an allergic reaction."
      }, {
        "code": "EUH205",
        "statement": "Contains epoxy constituents. May produce an allergic reaction."
      }, {
        "code": "EUH206",
        "statement": "Warning! Do not use together with other products. May release dangerous gases (chlorine)."
      }, {
        "code": "EUH207",
        "statement": "Warning! Contains cadmium. Dangerous fumes are formed during use. See information supplied by the manufacturer. Comply with the safety instructions."
      }, {
        "code": "EUH208",
        "statement": "Contains <name of sensitising substance>. May produce an allergic reaction."
      }, {
        "code": "EUH209",
        "statement": "Can become highly flammable in use."
      }, {
        "code": "EUH209A",
        "statement": "Can become flammable in use."
      }, {
        "code": "EUH210",
        "statement": "Safety data sheet available on request."
      }, {
        "code": "EUH401",
        "statement": "To avoid risks to human health and the environment, comply with the instructions for use."
      }]};

    function getHStatementsObject() {
      const result = {};
      ghs.hStatements.forEach(h => {
        if (result[h.code]) {
          throw new Error(`This should never happen, duplice hCode: ${h.code}`);
        }
        result[h.code] = h;
      });
      return result;
    }

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    function commonjsRequire(path) {
    	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
    }

    var jsonpath = {exports: {}};

    /*! jsonpath 1.1.1 */
    (function (module, exports$1) {
      (function (f) {
        {
          module.exports = f();
        }
      })(function () {
        return function e(t, n, r) {
          function s(o, u) {
            if (!n[o]) {
              if (!t[o]) {
                var a = typeof commonjsRequire == "function" && commonjsRequire;
                if (!u && a) return a(o, true);
                if (i) return i(o, true);
                var f = new Error("Cannot find module '" + o + "'");
                throw f.code = "MODULE_NOT_FOUND", f;
              }
              var l = n[o] = {
                exports: {}
              };
              t[o][0].call(l.exports, function (e) {
                var n = t[o][1][e];
                return s(n ? n : e);
              }, l, l.exports, e, t, n, r);
            }
            return n[o].exports;
          }
          var i = typeof commonjsRequire == "function" && commonjsRequire;
          for (var o = 0; o < r.length; o++) s(r[o]);
          return s;
        }({
          "./aesprim": [function (require, module, exports$1) {
            /*
              Copyright (C) 2013 Ariya Hidayat <ariya.hidayat@gmail.com>
              Copyright (C) 2013 Thaddee Tyl <thaddee.tyl@gmail.com>
              Copyright (C) 2013 Mathias Bynens <mathias@qiwi.be>
              Copyright (C) 2012 Ariya Hidayat <ariya.hidayat@gmail.com>
              Copyright (C) 2012 Mathias Bynens <mathias@qiwi.be>
              Copyright (C) 2012 Joost-Wim Boekesteijn <joost-wim@boekesteijn.nl>
              Copyright (C) 2012 Kris Kowal <kris.kowal@cixar.com>
              Copyright (C) 2012 Yusuke Suzuki <utatane.tea@gmail.com>
              Copyright (C) 2012 Arpad Borsos <arpad.borsos@googlemail.com>
              Copyright (C) 2011 Ariya Hidayat <ariya.hidayat@gmail.com>
            	  Redistribution and use in source and binary forms, with or without
              modification, are permitted provided that the following conditions are met:
            	    * Redistributions of source code must retain the above copyright
                  notice, this list of conditions and the following disclaimer.
                * Redistributions in binary form must reproduce the above copyright
                  notice, this list of conditions and the following disclaimer in the
                  documentation and/or other materials provided with the distribution.
            	  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
              AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
              IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
              ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
              DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
              (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
              LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
              ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
              (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
              THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
            */

            /*jslint bitwise:true plusplus:true */
            /*global esprima:true, define:true, exports:true, window: true,
            throwErrorTolerant: true,
            throwError: true, generateStatement: true, peek: true,
            parseAssignmentExpression: true, parseBlock: true, parseExpression: true,
            parseFunctionDeclaration: true, parseFunctionExpression: true,
            parseFunctionSourceElements: true, parseVariableIdentifier: true,
            parseLeftHandSideExpression: true,
            parseUnaryExpression: true,
            parseStatement: true, parseSourceElement: true */

            (function (root, factory) {

              // Universal Module Definition (UMD) to support AMD, CommonJS/Node.js,
              // Rhino, and plain browser loading.

              /* istanbul ignore next */
              if (typeof exports$1 !== 'undefined') {
                factory(exports$1);
              } else {
                factory(root.esprima = {});
              }
            })(this, function (exports$1) {

              var Token, TokenName, FnExprTokens, Syntax, PropertyKind, Messages, Regex, SyntaxTreeDelegate, source, strict, index, lineNumber, lineStart, length, delegate, lookahead, state, extra;
              Token = {
                BooleanLiteral: 1,
                EOF: 2,
                Identifier: 3,
                Keyword: 4,
                NullLiteral: 5,
                NumericLiteral: 6,
                Punctuator: 7,
                StringLiteral: 8,
                RegularExpression: 9
              };
              TokenName = {};
              TokenName[Token.BooleanLiteral] = 'Boolean';
              TokenName[Token.EOF] = '<end>';
              TokenName[Token.Identifier] = 'Identifier';
              TokenName[Token.Keyword] = 'Keyword';
              TokenName[Token.NullLiteral] = 'Null';
              TokenName[Token.NumericLiteral] = 'Numeric';
              TokenName[Token.Punctuator] = 'Punctuator';
              TokenName[Token.StringLiteral] = 'String';
              TokenName[Token.RegularExpression] = 'RegularExpression';

              // A function following one of those tokens is an expression.
              FnExprTokens = ['(', '{', '[', 'in', 'typeof', 'instanceof', 'new', 'return', 'case', 'delete', 'throw', 'void',
              // assignment operators
              '=', '+=', '-=', '*=', '/=', '%=', '<<=', '>>=', '>>>=', '&=', '|=', '^=', ',',
              // binary/unary operators
              '+', '-', '*', '/', '%', '++', '--', '<<', '>>', '>>>', '&', '|', '^', '!', '~', '&&', '||', '?', ':', '===', '==', '>=', '<=', '<', '>', '!=', '!=='];
              Syntax = {
                AssignmentExpression: 'AssignmentExpression',
                ArrayExpression: 'ArrayExpression',
                BlockStatement: 'BlockStatement',
                BinaryExpression: 'BinaryExpression',
                BreakStatement: 'BreakStatement',
                CallExpression: 'CallExpression',
                CatchClause: 'CatchClause',
                ConditionalExpression: 'ConditionalExpression',
                ContinueStatement: 'ContinueStatement',
                DoWhileStatement: 'DoWhileStatement',
                DebuggerStatement: 'DebuggerStatement',
                EmptyStatement: 'EmptyStatement',
                ExpressionStatement: 'ExpressionStatement',
                ForStatement: 'ForStatement',
                ForInStatement: 'ForInStatement',
                FunctionDeclaration: 'FunctionDeclaration',
                FunctionExpression: 'FunctionExpression',
                Identifier: 'Identifier',
                IfStatement: 'IfStatement',
                Literal: 'Literal',
                LabeledStatement: 'LabeledStatement',
                LogicalExpression: 'LogicalExpression',
                MemberExpression: 'MemberExpression',
                NewExpression: 'NewExpression',
                ObjectExpression: 'ObjectExpression',
                Program: 'Program',
                Property: 'Property',
                ReturnStatement: 'ReturnStatement',
                SequenceExpression: 'SequenceExpression',
                SwitchStatement: 'SwitchStatement',
                SwitchCase: 'SwitchCase',
                ThisExpression: 'ThisExpression',
                ThrowStatement: 'ThrowStatement',
                TryStatement: 'TryStatement',
                UnaryExpression: 'UnaryExpression',
                UpdateExpression: 'UpdateExpression',
                VariableDeclaration: 'VariableDeclaration',
                VariableDeclarator: 'VariableDeclarator',
                WhileStatement: 'WhileStatement',
                WithStatement: 'WithStatement'
              };
              PropertyKind = {
                Data: 1,
                Get: 2,
                Set: 4
              };

              // Error messages should be identical to V8.
              Messages = {
                UnexpectedToken: 'Unexpected token %0',
                UnexpectedNumber: 'Unexpected number',
                UnexpectedString: 'Unexpected string',
                UnexpectedIdentifier: 'Unexpected identifier',
                UnexpectedReserved: 'Unexpected reserved word',
                UnexpectedEOS: 'Unexpected end of input',
                NewlineAfterThrow: 'Illegal newline after throw',
                InvalidRegExp: 'Invalid regular expression',
                UnterminatedRegExp: 'Invalid regular expression: missing /',
                InvalidLHSInAssignment: 'Invalid left-hand side in assignment',
                InvalidLHSInForIn: 'Invalid left-hand side in for-in',
                MultipleDefaultsInSwitch: 'More than one default clause in switch statement',
                NoCatchOrFinally: 'Missing catch or finally after try',
                UnknownLabel: 'Undefined label \'%0\'',
                Redeclaration: '%0 \'%1\' has already been declared',
                IllegalContinue: 'Illegal continue statement',
                IllegalBreak: 'Illegal break statement',
                IllegalReturn: 'Illegal return statement',
                StrictModeWith: 'Strict mode code may not include a with statement',
                StrictCatchVariable: 'Catch variable may not be eval or arguments in strict mode',
                StrictVarName: 'Variable name may not be eval or arguments in strict mode',
                StrictParamName: 'Parameter name eval or arguments is not allowed in strict mode',
                StrictParamDupe: 'Strict mode function may not have duplicate parameter names',
                StrictFunctionName: 'Function name may not be eval or arguments in strict mode',
                StrictOctalLiteral: 'Octal literals are not allowed in strict mode.',
                StrictDelete: 'Delete of an unqualified identifier in strict mode.',
                StrictDuplicateProperty: 'Duplicate data property in object literal not allowed in strict mode',
                AccessorDataProperty: 'Object literal may not have data and accessor property with the same name',
                AccessorGetSet: 'Object literal may not have multiple get/set accessors with the same name',
                StrictLHSAssignment: 'Assignment to eval or arguments is not allowed in strict mode',
                StrictLHSPostfix: 'Postfix increment/decrement may not have eval or arguments operand in strict mode',
                StrictLHSPrefix: 'Prefix increment/decrement may not have eval or arguments operand in strict mode',
                StrictReservedWord: 'Use of future reserved word in strict mode'
              };

              // See also tools/generate-unicode-regex.py.
              Regex = {
                NonAsciiIdentifierStart: new RegExp('[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0620-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u08A0\u08A2-\u08AC\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971-\u0977\u0979-\u097F\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D60\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1CF5\u1CF6\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005-\u3007\u3021-\u3029\u3031-\u3035\u3038-\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA66E\uA67F-\uA697\uA6A0-\uA6EF\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]'),
                NonAsciiIdentifierPart: new RegExp('[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0300-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u0483-\u0487\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u0591-\u05BD\u05BF\u05C1\u05C2\u05C4\u05C5\u05C7\u05D0-\u05EA\u05F0-\u05F2\u0610-\u061A\u0620-\u0669\u066E-\u06D3\u06D5-\u06DC\u06DF-\u06E8\u06EA-\u06FC\u06FF\u0710-\u074A\u074D-\u07B1\u07C0-\u07F5\u07FA\u0800-\u082D\u0840-\u085B\u08A0\u08A2-\u08AC\u08E4-\u08FE\u0900-\u0963\u0966-\u096F\u0971-\u0977\u0979-\u097F\u0981-\u0983\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BC-\u09C4\u09C7\u09C8\u09CB-\u09CE\u09D7\u09DC\u09DD\u09DF-\u09E3\u09E6-\u09F1\u0A01-\u0A03\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A3C\u0A3E-\u0A42\u0A47\u0A48\u0A4B-\u0A4D\u0A51\u0A59-\u0A5C\u0A5E\u0A66-\u0A75\u0A81-\u0A83\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABC-\u0AC5\u0AC7-\u0AC9\u0ACB-\u0ACD\u0AD0\u0AE0-\u0AE3\u0AE6-\u0AEF\u0B01-\u0B03\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3C-\u0B44\u0B47\u0B48\u0B4B-\u0B4D\u0B56\u0B57\u0B5C\u0B5D\u0B5F-\u0B63\u0B66-\u0B6F\u0B71\u0B82\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BBE-\u0BC2\u0BC6-\u0BC8\u0BCA-\u0BCD\u0BD0\u0BD7\u0BE6-\u0BEF\u0C01-\u0C03\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D-\u0C44\u0C46-\u0C48\u0C4A-\u0C4D\u0C55\u0C56\u0C58\u0C59\u0C60-\u0C63\u0C66-\u0C6F\u0C82\u0C83\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBC-\u0CC4\u0CC6-\u0CC8\u0CCA-\u0CCD\u0CD5\u0CD6\u0CDE\u0CE0-\u0CE3\u0CE6-\u0CEF\u0CF1\u0CF2\u0D02\u0D03\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D-\u0D44\u0D46-\u0D48\u0D4A-\u0D4E\u0D57\u0D60-\u0D63\u0D66-\u0D6F\u0D7A-\u0D7F\u0D82\u0D83\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0DCA\u0DCF-\u0DD4\u0DD6\u0DD8-\u0DDF\u0DF2\u0DF3\u0E01-\u0E3A\u0E40-\u0E4E\u0E50-\u0E59\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB9\u0EBB-\u0EBD\u0EC0-\u0EC4\u0EC6\u0EC8-\u0ECD\u0ED0-\u0ED9\u0EDC-\u0EDF\u0F00\u0F18\u0F19\u0F20-\u0F29\u0F35\u0F37\u0F39\u0F3E-\u0F47\u0F49-\u0F6C\u0F71-\u0F84\u0F86-\u0F97\u0F99-\u0FBC\u0FC6\u1000-\u1049\u1050-\u109D\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u135D-\u135F\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1714\u1720-\u1734\u1740-\u1753\u1760-\u176C\u176E-\u1770\u1772\u1773\u1780-\u17D3\u17D7\u17DC\u17DD\u17E0-\u17E9\u180B-\u180D\u1810-\u1819\u1820-\u1877\u1880-\u18AA\u18B0-\u18F5\u1900-\u191C\u1920-\u192B\u1930-\u193B\u1946-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u19D0-\u19D9\u1A00-\u1A1B\u1A20-\u1A5E\u1A60-\u1A7C\u1A7F-\u1A89\u1A90-\u1A99\u1AA7\u1B00-\u1B4B\u1B50-\u1B59\u1B6B-\u1B73\u1B80-\u1BF3\u1C00-\u1C37\u1C40-\u1C49\u1C4D-\u1C7D\u1CD0-\u1CD2\u1CD4-\u1CF6\u1D00-\u1DE6\u1DFC-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u200C\u200D\u203F\u2040\u2054\u2071\u207F\u2090-\u209C\u20D0-\u20DC\u20E1\u20E5-\u20F0\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D7F-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2DE0-\u2DFF\u2E2F\u3005-\u3007\u3021-\u302F\u3031-\u3035\u3038-\u303C\u3041-\u3096\u3099\u309A\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA62B\uA640-\uA66F\uA674-\uA67D\uA67F-\uA697\uA69F-\uA6F1\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA827\uA840-\uA873\uA880-\uA8C4\uA8D0-\uA8D9\uA8E0-\uA8F7\uA8FB\uA900-\uA92D\uA930-\uA953\uA960-\uA97C\uA980-\uA9C0\uA9CF-\uA9D9\uAA00-\uAA36\uAA40-\uAA4D\uAA50-\uAA59\uAA60-\uAA76\uAA7A\uAA7B\uAA80-\uAAC2\uAADB-\uAADD\uAAE0-\uAAEF\uAAF2-\uAAF6\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABEA\uABEC\uABED\uABF0-\uABF9\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE00-\uFE0F\uFE20-\uFE26\uFE33\uFE34\uFE4D-\uFE4F\uFE70-\uFE74\uFE76-\uFEFC\uFF10-\uFF19\uFF21-\uFF3A\uFF3F\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]')
              };

              // Ensure the condition is true, otherwise throw an error.
              // This is only to have a better contract semantic, i.e. another safety net
              // to catch a logic error. The condition shall be fulfilled in normal case.
              // Do NOT use this to enforce a certain condition on any user input.

              function assert(condition, message) {
                /* istanbul ignore if */
                if (!condition) {
                  throw new Error('ASSERT: ' + message);
                }
              }
              function isDecimalDigit(ch) {
                return ch >= 48 && ch <= 57; // 0..9
              }
              function isHexDigit(ch) {
                return '0123456789abcdefABCDEF'.indexOf(ch) >= 0;
              }
              function isOctalDigit(ch) {
                return '01234567'.indexOf(ch) >= 0;
              }

              // 7.2 White Space

              function isWhiteSpace(ch) {
                return ch === 0x20 || ch === 0x09 || ch === 0x0B || ch === 0x0C || ch === 0xA0 || ch >= 0x1680 && [0x1680, 0x180E, 0x2000, 0x2001, 0x2002, 0x2003, 0x2004, 0x2005, 0x2006, 0x2007, 0x2008, 0x2009, 0x200A, 0x202F, 0x205F, 0x3000, 0xFEFF].indexOf(ch) >= 0;
              }

              // 7.3 Line Terminators

              function isLineTerminator(ch) {
                return ch === 0x0A || ch === 0x0D || ch === 0x2028 || ch === 0x2029;
              }

              // 7.6 Identifier Names and Identifiers

              function isIdentifierStart(ch) {
                return ch == 0x40 || ch === 0x24 || ch === 0x5F ||
                // $ (dollar) and _ (underscore)
                ch >= 0x41 && ch <= 0x5A ||
                // A..Z
                ch >= 0x61 && ch <= 0x7A ||
                // a..z
                ch === 0x5C ||
                // \ (backslash)
                ch >= 0x80 && Regex.NonAsciiIdentifierStart.test(String.fromCharCode(ch));
              }
              function isIdentifierPart(ch) {
                return ch === 0x24 || ch === 0x5F ||
                // $ (dollar) and _ (underscore)
                ch >= 0x41 && ch <= 0x5A ||
                // A..Z
                ch >= 0x61 && ch <= 0x7A ||
                // a..z
                ch >= 0x30 && ch <= 0x39 ||
                // 0..9
                ch === 0x5C ||
                // \ (backslash)
                ch >= 0x80 && Regex.NonAsciiIdentifierPart.test(String.fromCharCode(ch));
              }

              // 7.6.1.2 Future Reserved Words

              function isFutureReservedWord(id) {
                switch (id) {
                  case 'class':
                  case 'enum':
                  case 'export':
                  case 'extends':
                  case 'import':
                  case 'super':
                    return true;
                  default:
                    return false;
                }
              }
              function isStrictModeReservedWord(id) {
                switch (id) {
                  case 'implements':
                  case 'interface':
                  case 'package':
                  case 'private':
                  case 'protected':
                  case 'public':
                  case 'static':
                  case 'yield':
                  case 'let':
                    return true;
                  default:
                    return false;
                }
              }
              function isRestrictedWord(id) {
                return id === 'eval' || id === 'arguments';
              }

              // 7.6.1.1 Keywords

              function isKeyword(id) {
                if (strict && isStrictModeReservedWord(id)) {
                  return true;
                }

                // 'const' is specialized as Keyword in V8.
                // 'yield' and 'let' are for compatiblity with SpiderMonkey and ES.next.
                // Some others are from future reserved words.

                switch (id.length) {
                  case 2:
                    return id === 'if' || id === 'in' || id === 'do';
                  case 3:
                    return id === 'var' || id === 'for' || id === 'new' || id === 'try' || id === 'let';
                  case 4:
                    return id === 'this' || id === 'else' || id === 'case' || id === 'void' || id === 'with' || id === 'enum';
                  case 5:
                    return id === 'while' || id === 'break' || id === 'catch' || id === 'throw' || id === 'const' || id === 'yield' || id === 'class' || id === 'super';
                  case 6:
                    return id === 'return' || id === 'typeof' || id === 'delete' || id === 'switch' || id === 'export' || id === 'import';
                  case 7:
                    return id === 'default' || id === 'finally' || id === 'extends';
                  case 8:
                    return id === 'function' || id === 'continue' || id === 'debugger';
                  case 10:
                    return id === 'instanceof';
                  default:
                    return false;
                }
              }

              // 7.4 Comments

              function addComment(type, value, start, end, loc) {
                var comment;
                assert(typeof start === 'number', 'Comment must have valid position');

                // Because the way the actual token is scanned, often the comments
                // (if any) are skipped twice during the lexical analysis.
                // Thus, we need to skip adding a comment if the comment array already
                // handled it.
                if (state.lastCommentStart >= start) {
                  return;
                }
                state.lastCommentStart = start;
                comment = {
                  type: type,
                  value: value
                };
                if (extra.range) {
                  comment.range = [start, end];
                }
                if (extra.loc) {
                  comment.loc = loc;
                }
                extra.comments.push(comment);
                if (extra.attachComment) {
                  extra.leadingComments.push(comment);
                  extra.trailingComments.push(comment);
                }
              }
              function skipSingleLineComment(offset) {
                var start, loc, ch, comment;
                start = index - offset;
                loc = {
                  start: {
                    line: lineNumber,
                    column: index - lineStart - offset
                  }
                };
                while (index < length) {
                  ch = source.charCodeAt(index);
                  ++index;
                  if (isLineTerminator(ch)) {
                    if (extra.comments) {
                      comment = source.slice(start + offset, index - 1);
                      loc.end = {
                        line: lineNumber,
                        column: index - lineStart - 1
                      };
                      addComment('Line', comment, start, index - 1, loc);
                    }
                    if (ch === 13 && source.charCodeAt(index) === 10) {
                      ++index;
                    }
                    ++lineNumber;
                    lineStart = index;
                    return;
                  }
                }
                if (extra.comments) {
                  comment = source.slice(start + offset, index);
                  loc.end = {
                    line: lineNumber,
                    column: index - lineStart
                  };
                  addComment('Line', comment, start, index, loc);
                }
              }
              function skipMultiLineComment() {
                var start, loc, ch, comment;
                if (extra.comments) {
                  start = index - 2;
                  loc = {
                    start: {
                      line: lineNumber,
                      column: index - lineStart - 2
                    }
                  };
                }
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (isLineTerminator(ch)) {
                    if (ch === 0x0D && source.charCodeAt(index + 1) === 0x0A) {
                      ++index;
                    }
                    ++lineNumber;
                    ++index;
                    lineStart = index;
                    if (index >= length) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                  } else if (ch === 0x2A) {
                    // Block comment ends with '*/'.
                    if (source.charCodeAt(index + 1) === 0x2F) {
                      ++index;
                      ++index;
                      if (extra.comments) {
                        comment = source.slice(start + 2, index - 2);
                        loc.end = {
                          line: lineNumber,
                          column: index - lineStart
                        };
                        addComment('Block', comment, start, index, loc);
                      }
                      return;
                    }
                    ++index;
                  } else {
                    ++index;
                  }
                }
                throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
              }
              function skipComment() {
                var ch, start;
                start = index === 0;
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (isWhiteSpace(ch)) {
                    ++index;
                  } else if (isLineTerminator(ch)) {
                    ++index;
                    if (ch === 0x0D && source.charCodeAt(index) === 0x0A) {
                      ++index;
                    }
                    ++lineNumber;
                    lineStart = index;
                    start = true;
                  } else if (ch === 0x2F) {
                    // U+002F is '/'
                    ch = source.charCodeAt(index + 1);
                    if (ch === 0x2F) {
                      ++index;
                      ++index;
                      skipSingleLineComment(2);
                      start = true;
                    } else if (ch === 0x2A) {
                      // U+002A is '*'
                      ++index;
                      ++index;
                      skipMultiLineComment();
                    } else {
                      break;
                    }
                  } else if (start && ch === 0x2D) {
                    // U+002D is '-'
                    // U+003E is '>'
                    if (source.charCodeAt(index + 1) === 0x2D && source.charCodeAt(index + 2) === 0x3E) {
                      // '-->' is a single-line comment
                      index += 3;
                      skipSingleLineComment(3);
                    } else {
                      break;
                    }
                  } else if (ch === 0x3C) {
                    // U+003C is '<'
                    if (source.slice(index + 1, index + 4) === '!--') {
                      ++index; // `<`
                      ++index; // `!`
                      ++index; // `-`
                      ++index; // `-`
                      skipSingleLineComment(4);
                    } else {
                      break;
                    }
                  } else {
                    break;
                  }
                }
              }
              function scanHexEscape(prefix) {
                var i,
                  len,
                  ch,
                  code = 0;
                len = prefix === 'u' ? 4 : 2;
                for (i = 0; i < len; ++i) {
                  if (index < length && isHexDigit(source[index])) {
                    ch = source[index++];
                    code = code * 16 + '0123456789abcdef'.indexOf(ch.toLowerCase());
                  } else {
                    return '';
                  }
                }
                return String.fromCharCode(code);
              }
              function getEscapedIdentifier() {
                var ch, id;
                ch = source.charCodeAt(index++);
                id = String.fromCharCode(ch);

                // '\u' (U+005C, U+0075) denotes an escaped character.
                if (ch === 0x5C) {
                  if (source.charCodeAt(index) !== 0x75) {
                    throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                  }
                  ++index;
                  ch = scanHexEscape('u');
                  if (!ch || ch === '\\' || !isIdentifierStart(ch.charCodeAt(0))) {
                    throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                  }
                  id = ch;
                }
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (!isIdentifierPart(ch)) {
                    break;
                  }
                  ++index;
                  id += String.fromCharCode(ch);

                  // '\u' (U+005C, U+0075) denotes an escaped character.
                  if (ch === 0x5C) {
                    id = id.substr(0, id.length - 1);
                    if (source.charCodeAt(index) !== 0x75) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                    ++index;
                    ch = scanHexEscape('u');
                    if (!ch || ch === '\\' || !isIdentifierPart(ch.charCodeAt(0))) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                    id += ch;
                  }
                }
                return id;
              }
              function getIdentifier() {
                var start, ch;
                start = index++;
                while (index < length) {
                  ch = source.charCodeAt(index);
                  if (ch === 0x5C) {
                    // Blackslash (U+005C) marks Unicode escape sequence.
                    index = start;
                    return getEscapedIdentifier();
                  }
                  if (isIdentifierPart(ch)) {
                    ++index;
                  } else {
                    break;
                  }
                }
                return source.slice(start, index);
              }
              function scanIdentifier() {
                var start, id, type;
                start = index;

                // Backslash (U+005C) starts an escaped character.
                id = source.charCodeAt(index) === 0x5C ? getEscapedIdentifier() : getIdentifier();

                // There is no keyword or literal with only one character.
                // Thus, it must be an identifier.
                if (id.length === 1) {
                  type = Token.Identifier;
                } else if (isKeyword(id)) {
                  type = Token.Keyword;
                } else if (id === 'null') {
                  type = Token.NullLiteral;
                } else if (id === 'true' || id === 'false') {
                  type = Token.BooleanLiteral;
                } else {
                  type = Token.Identifier;
                }
                return {
                  type: type,
                  value: id,
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }

              // 7.7 Punctuators

              function scanPunctuator() {
                var start = index,
                  code = source.charCodeAt(index),
                  code2,
                  ch1 = source[index],
                  ch2,
                  ch3,
                  ch4;
                switch (code) {
                  // Check for most common single-character punctuators.
                  case 0x2E: // . dot
                  case 0x28: // ( open bracket
                  case 0x29: // ) close bracket
                  case 0x3B: // ; semicolon
                  case 0x2C: // , comma
                  case 0x7B: // { open curly brace
                  case 0x7D: // } close curly brace
                  case 0x5B: // [
                  case 0x5D: // ]
                  case 0x3A: // :
                  case 0x3F: // ?
                  case 0x7E:
                    // ~
                    ++index;
                    if (extra.tokenize) {
                      if (code === 0x28) {
                        extra.openParenToken = extra.tokens.length;
                      } else if (code === 0x7B) {
                        extra.openCurlyToken = extra.tokens.length;
                      }
                    }
                    return {
                      type: Token.Punctuator,
                      value: String.fromCharCode(code),
                      lineNumber: lineNumber,
                      lineStart: lineStart,
                      start: start,
                      end: index
                    };
                  default:
                    code2 = source.charCodeAt(index + 1);

                    // '=' (U+003D) marks an assignment or comparison operator.
                    if (code2 === 0x3D) {
                      switch (code) {
                        case 0x2B: // +
                        case 0x2D: // -
                        case 0x2F: // /
                        case 0x3C: // <
                        case 0x3E: // >
                        case 0x5E: // ^
                        case 0x7C: // |
                        case 0x25: // %
                        case 0x26: // &
                        case 0x2A:
                          // *
                          index += 2;
                          return {
                            type: Token.Punctuator,
                            value: String.fromCharCode(code) + String.fromCharCode(code2),
                            lineNumber: lineNumber,
                            lineStart: lineStart,
                            start: start,
                            end: index
                          };
                        case 0x21: // !
                        case 0x3D:
                          // =
                          index += 2;

                          // !== and ===
                          if (source.charCodeAt(index) === 0x3D) {
                            ++index;
                          }
                          return {
                            type: Token.Punctuator,
                            value: source.slice(start, index),
                            lineNumber: lineNumber,
                            lineStart: lineStart,
                            start: start,
                            end: index
                          };
                      }
                    }
                }

                // 4-character punctuator: >>>=

                ch4 = source.substr(index, 4);
                if (ch4 === '>>>=') {
                  index += 4;
                  return {
                    type: Token.Punctuator,
                    value: ch4,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }

                // 3-character punctuators: === !== >>> <<= >>=

                ch3 = ch4.substr(0, 3);
                if (ch3 === '>>>' || ch3 === '<<=' || ch3 === '>>=') {
                  index += 3;
                  return {
                    type: Token.Punctuator,
                    value: ch3,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }

                // Other 2-character punctuators: ++ -- << >> && ||
                ch2 = ch3.substr(0, 2);
                if (ch1 === ch2[1] && '+-<>&|'.indexOf(ch1) >= 0 || ch2 === '=>') {
                  index += 2;
                  return {
                    type: Token.Punctuator,
                    value: ch2,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }

                // 1-character punctuators: < > = ! + - * % & | ^ /
                if ('<>=!+-*%&|^/'.indexOf(ch1) >= 0) {
                  ++index;
                  return {
                    type: Token.Punctuator,
                    value: ch1,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }
                throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
              }

              // 7.8.3 Numeric Literals

              function scanHexLiteral(start) {
                var number = '';
                while (index < length) {
                  if (!isHexDigit(source[index])) {
                    break;
                  }
                  number += source[index++];
                }
                if (number.length === 0) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                if (isIdentifierStart(source.charCodeAt(index))) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.NumericLiteral,
                  value: parseInt('0x' + number, 16),
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }
              function scanOctalLiteral(start) {
                var number = '0' + source[index++];
                while (index < length) {
                  if (!isOctalDigit(source[index])) {
                    break;
                  }
                  number += source[index++];
                }
                if (isIdentifierStart(source.charCodeAt(index)) || isDecimalDigit(source.charCodeAt(index))) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.NumericLiteral,
                  value: parseInt(number, 8),
                  octal: true,
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }
              function scanNumericLiteral() {
                var number, start, ch;
                ch = source[index];
                assert(isDecimalDigit(ch.charCodeAt(0)) || ch === '.', 'Numeric literal must start with a decimal digit or a decimal point');
                start = index;
                number = '';
                if (ch !== '.') {
                  number = source[index++];
                  ch = source[index];

                  // Hex number starts with '0x'.
                  // Octal number starts with '0'.
                  if (number === '0') {
                    if (ch === 'x' || ch === 'X') {
                      ++index;
                      return scanHexLiteral(start);
                    }
                    if (isOctalDigit(ch)) {
                      return scanOctalLiteral(start);
                    }

                    // decimal number starts with '0' such as '09' is illegal.
                    if (ch && isDecimalDigit(ch.charCodeAt(0))) {
                      throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                  }
                  while (isDecimalDigit(source.charCodeAt(index))) {
                    number += source[index++];
                  }
                  ch = source[index];
                }
                if (ch === '.') {
                  number += source[index++];
                  while (isDecimalDigit(source.charCodeAt(index))) {
                    number += source[index++];
                  }
                  ch = source[index];
                }
                if (ch === 'e' || ch === 'E') {
                  number += source[index++];
                  ch = source[index];
                  if (ch === '+' || ch === '-') {
                    number += source[index++];
                  }
                  if (isDecimalDigit(source.charCodeAt(index))) {
                    while (isDecimalDigit(source.charCodeAt(index))) {
                      number += source[index++];
                    }
                  } else {
                    throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                  }
                }
                if (isIdentifierStart(source.charCodeAt(index))) {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.NumericLiteral,
                  value: parseFloat(number),
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }

              // 7.8.4 String Literals

              function scanStringLiteral() {
                var str = '',
                  quote,
                  start,
                  ch,
                  code,
                  unescaped,
                  restore,
                  octal = false,
                  startLineNumber,
                  startLineStart;
                startLineNumber = lineNumber;
                startLineStart = lineStart;
                quote = source[index];
                assert(quote === '\'' || quote === '"', 'String literal must starts with a quote');
                start = index;
                ++index;
                while (index < length) {
                  ch = source[index++];
                  if (ch === quote) {
                    quote = '';
                    break;
                  } else if (ch === '\\') {
                    ch = source[index++];
                    if (!ch || !isLineTerminator(ch.charCodeAt(0))) {
                      switch (ch) {
                        case 'u':
                        case 'x':
                          restore = index;
                          unescaped = scanHexEscape(ch);
                          if (unescaped) {
                            str += unescaped;
                          } else {
                            index = restore;
                            str += ch;
                          }
                          break;
                        case 'n':
                          str += '\n';
                          break;
                        case 'r':
                          str += '\r';
                          break;
                        case 't':
                          str += '\t';
                          break;
                        case 'b':
                          str += '\b';
                          break;
                        case 'f':
                          str += '\f';
                          break;
                        case 'v':
                          str += '\x0B';
                          break;
                        default:
                          if (isOctalDigit(ch)) {
                            code = '01234567'.indexOf(ch);

                            // \0 is not octal escape sequence
                            if (code !== 0) {
                              octal = true;
                            }
                            if (index < length && isOctalDigit(source[index])) {
                              octal = true;
                              code = code * 8 + '01234567'.indexOf(source[index++]);

                              // 3 digits are only allowed when string starts
                              // with 0, 1, 2, 3
                              if ('0123'.indexOf(ch) >= 0 && index < length && isOctalDigit(source[index])) {
                                code = code * 8 + '01234567'.indexOf(source[index++]);
                              }
                            }
                            str += String.fromCharCode(code);
                          } else {
                            str += ch;
                          }
                          break;
                      }
                    } else {
                      ++lineNumber;
                      if (ch === '\r' && source[index] === '\n') {
                        ++index;
                      }
                      lineStart = index;
                    }
                  } else if (isLineTerminator(ch.charCodeAt(0))) {
                    break;
                  } else {
                    str += ch;
                  }
                }
                if (quote !== '') {
                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
                }
                return {
                  type: Token.StringLiteral,
                  value: str,
                  octal: octal,
                  startLineNumber: startLineNumber,
                  startLineStart: startLineStart,
                  lineNumber: lineNumber,
                  lineStart: lineStart,
                  start: start,
                  end: index
                };
              }
              function testRegExp(pattern, flags) {
                var value;
                try {
                  value = new RegExp(pattern, flags);
                } catch (e) {
                  throwError({}, Messages.InvalidRegExp);
                }
                return value;
              }
              function scanRegExpBody() {
                var ch, str, classMarker, terminated, body;
                ch = source[index];
                assert(ch === '/', 'Regular expression literal must start with a slash');
                str = source[index++];
                classMarker = false;
                terminated = false;
                while (index < length) {
                  ch = source[index++];
                  str += ch;
                  if (ch === '\\') {
                    ch = source[index++];
                    // ECMA-262 7.8.5
                    if (isLineTerminator(ch.charCodeAt(0))) {
                      throwError({}, Messages.UnterminatedRegExp);
                    }
                    str += ch;
                  } else if (isLineTerminator(ch.charCodeAt(0))) {
                    throwError({}, Messages.UnterminatedRegExp);
                  } else if (classMarker) {
                    if (ch === ']') {
                      classMarker = false;
                    }
                  } else {
                    if (ch === '/') {
                      terminated = true;
                      break;
                    } else if (ch === '[') {
                      classMarker = true;
                    }
                  }
                }
                if (!terminated) {
                  throwError({}, Messages.UnterminatedRegExp);
                }

                // Exclude leading and trailing slash.
                body = str.substr(1, str.length - 2);
                return {
                  value: body,
                  literal: str
                };
              }
              function scanRegExpFlags() {
                var ch, str, flags, restore;
                str = '';
                flags = '';
                while (index < length) {
                  ch = source[index];
                  if (!isIdentifierPart(ch.charCodeAt(0))) {
                    break;
                  }
                  ++index;
                  if (ch === '\\' && index < length) {
                    ch = source[index];
                    if (ch === 'u') {
                      ++index;
                      restore = index;
                      ch = scanHexEscape('u');
                      if (ch) {
                        flags += ch;
                        for (str += '\\u'; restore < index; ++restore) {
                          str += source[restore];
                        }
                      } else {
                        index = restore;
                        flags += 'u';
                        str += '\\u';
                      }
                      throwErrorTolerant({}, Messages.UnexpectedToken, 'ILLEGAL');
                    } else {
                      str += '\\';
                      throwErrorTolerant({}, Messages.UnexpectedToken, 'ILLEGAL');
                    }
                  } else {
                    flags += ch;
                    str += ch;
                  }
                }
                return {
                  value: flags,
                  literal: str
                };
              }
              function scanRegExp() {
                var start, body, flags, value;
                lookahead = null;
                skipComment();
                start = index;
                body = scanRegExpBody();
                flags = scanRegExpFlags();
                value = testRegExp(body.value, flags.value);
                if (extra.tokenize) {
                  return {
                    type: Token.RegularExpression,
                    value: value,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: start,
                    end: index
                  };
                }
                return {
                  literal: body.literal + flags.literal,
                  value: value,
                  start: start,
                  end: index
                };
              }
              function collectRegex() {
                var pos, loc, regex, token;
                skipComment();
                pos = index;
                loc = {
                  start: {
                    line: lineNumber,
                    column: index - lineStart
                  }
                };
                regex = scanRegExp();
                loc.end = {
                  line: lineNumber,
                  column: index - lineStart
                };

                /* istanbul ignore next */
                if (!extra.tokenize) {
                  // Pop the previous token, which is likely '/' or '/='
                  if (extra.tokens.length > 0) {
                    token = extra.tokens[extra.tokens.length - 1];
                    if (token.range[0] === pos && token.type === 'Punctuator') {
                      if (token.value === '/' || token.value === '/=') {
                        extra.tokens.pop();
                      }
                    }
                  }
                  extra.tokens.push({
                    type: 'RegularExpression',
                    value: regex.literal,
                    range: [pos, index],
                    loc: loc
                  });
                }
                return regex;
              }
              function isIdentifierName(token) {
                return token.type === Token.Identifier || token.type === Token.Keyword || token.type === Token.BooleanLiteral || token.type === Token.NullLiteral;
              }
              function advanceSlash() {
                var prevToken, checkToken;
                // Using the following algorithm:
                // https://github.com/mozilla/sweet.js/wiki/design
                prevToken = extra.tokens[extra.tokens.length - 1];
                if (!prevToken) {
                  // Nothing before that: it cannot be a division.
                  return collectRegex();
                }
                if (prevToken.type === 'Punctuator') {
                  if (prevToken.value === ']') {
                    return scanPunctuator();
                  }
                  if (prevToken.value === ')') {
                    checkToken = extra.tokens[extra.openParenToken - 1];
                    if (checkToken && checkToken.type === 'Keyword' && (checkToken.value === 'if' || checkToken.value === 'while' || checkToken.value === 'for' || checkToken.value === 'with')) {
                      return collectRegex();
                    }
                    return scanPunctuator();
                  }
                  if (prevToken.value === '}') {
                    // Dividing a function by anything makes little sense,
                    // but we have to check for that.
                    if (extra.tokens[extra.openCurlyToken - 3] && extra.tokens[extra.openCurlyToken - 3].type === 'Keyword') {
                      // Anonymous function.
                      checkToken = extra.tokens[extra.openCurlyToken - 4];
                      if (!checkToken) {
                        return scanPunctuator();
                      }
                    } else if (extra.tokens[extra.openCurlyToken - 4] && extra.tokens[extra.openCurlyToken - 4].type === 'Keyword') {
                      // Named function.
                      checkToken = extra.tokens[extra.openCurlyToken - 5];
                      if (!checkToken) {
                        return collectRegex();
                      }
                    } else {
                      return scanPunctuator();
                    }
                    // checkToken determines whether the function is
                    // a declaration or an expression.
                    if (FnExprTokens.indexOf(checkToken.value) >= 0) {
                      // It is an expression.
                      return scanPunctuator();
                    }
                    // It is a declaration.
                    return collectRegex();
                  }
                  return collectRegex();
                }
                if (prevToken.type === 'Keyword') {
                  return collectRegex();
                }
                return scanPunctuator();
              }
              function advance() {
                var ch;
                skipComment();
                if (index >= length) {
                  return {
                    type: Token.EOF,
                    lineNumber: lineNumber,
                    lineStart: lineStart,
                    start: index,
                    end: index
                  };
                }
                ch = source.charCodeAt(index);
                if (isIdentifierStart(ch)) {
                  return scanIdentifier();
                }

                // Very common: ( and ) and ;
                if (ch === 0x28 || ch === 0x29 || ch === 0x3B) {
                  return scanPunctuator();
                }

                // String literal starts with single quote (U+0027) or double quote (U+0022).
                if (ch === 0x27 || ch === 0x22) {
                  return scanStringLiteral();
                }

                // Dot (.) U+002E can also start a floating-point number, hence the need
                // to check the next character.
                if (ch === 0x2E) {
                  if (isDecimalDigit(source.charCodeAt(index + 1))) {
                    return scanNumericLiteral();
                  }
                  return scanPunctuator();
                }
                if (isDecimalDigit(ch)) {
                  return scanNumericLiteral();
                }

                // Slash (/) U+002F can also start a regex.
                if (extra.tokenize && ch === 0x2F) {
                  return advanceSlash();
                }
                return scanPunctuator();
              }
              function collectToken() {
                var loc, token, value;
                skipComment();
                loc = {
                  start: {
                    line: lineNumber,
                    column: index - lineStart
                  }
                };
                token = advance();
                loc.end = {
                  line: lineNumber,
                  column: index - lineStart
                };
                if (token.type !== Token.EOF) {
                  value = source.slice(token.start, token.end);
                  extra.tokens.push({
                    type: TokenName[token.type],
                    value: value,
                    range: [token.start, token.end],
                    loc: loc
                  });
                }
                return token;
              }
              function lex() {
                var token;
                token = lookahead;
                index = token.end;
                lineNumber = token.lineNumber;
                lineStart = token.lineStart;
                lookahead = typeof extra.tokens !== 'undefined' ? collectToken() : advance();
                index = token.end;
                lineNumber = token.lineNumber;
                lineStart = token.lineStart;
                return token;
              }
              function peek() {
                var pos, line, start;
                pos = index;
                line = lineNumber;
                start = lineStart;
                lookahead = typeof extra.tokens !== 'undefined' ? collectToken() : advance();
                index = pos;
                lineNumber = line;
                lineStart = start;
              }
              function Position(line, column) {
                this.line = line;
                this.column = column;
              }
              function SourceLocation(startLine, startColumn, line, column) {
                this.start = new Position(startLine, startColumn);
                this.end = new Position(line, column);
              }
              SyntaxTreeDelegate = {
                name: 'SyntaxTree',
                processComment: function (node) {
                  var lastChild, trailingComments;
                  if (node.type === Syntax.Program) {
                    if (node.body.length > 0) {
                      return;
                    }
                  }
                  if (extra.trailingComments.length > 0) {
                    if (extra.trailingComments[0].range[0] >= node.range[1]) {
                      trailingComments = extra.trailingComments;
                      extra.trailingComments = [];
                    } else {
                      extra.trailingComments.length = 0;
                    }
                  } else {
                    if (extra.bottomRightStack.length > 0 && extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments && extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments[0].range[0] >= node.range[1]) {
                      trailingComments = extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments;
                      delete extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments;
                    }
                  }

                  // Eating the stack.
                  while (extra.bottomRightStack.length > 0 && extra.bottomRightStack[extra.bottomRightStack.length - 1].range[0] >= node.range[0]) {
                    lastChild = extra.bottomRightStack.pop();
                  }
                  if (lastChild) {
                    if (lastChild.leadingComments && lastChild.leadingComments[lastChild.leadingComments.length - 1].range[1] <= node.range[0]) {
                      node.leadingComments = lastChild.leadingComments;
                      delete lastChild.leadingComments;
                    }
                  } else if (extra.leadingComments.length > 0 && extra.leadingComments[extra.leadingComments.length - 1].range[1] <= node.range[0]) {
                    node.leadingComments = extra.leadingComments;
                    extra.leadingComments = [];
                  }
                  if (trailingComments) {
                    node.trailingComments = trailingComments;
                  }
                  extra.bottomRightStack.push(node);
                },
                markEnd: function (node, startToken) {
                  if (extra.range) {
                    node.range = [startToken.start, index];
                  }
                  if (extra.loc) {
                    node.loc = new SourceLocation(startToken.startLineNumber === undefined ? startToken.lineNumber : startToken.startLineNumber, startToken.start - (startToken.startLineStart === undefined ? startToken.lineStart : startToken.startLineStart), lineNumber, index - lineStart);
                    this.postProcess(node);
                  }
                  if (extra.attachComment) {
                    this.processComment(node);
                  }
                  return node;
                },
                postProcess: function (node) {
                  if (extra.source) {
                    node.loc.source = extra.source;
                  }
                  return node;
                },
                createArrayExpression: function (elements) {
                  return {
                    type: Syntax.ArrayExpression,
                    elements: elements
                  };
                },
                createAssignmentExpression: function (operator, left, right) {
                  return {
                    type: Syntax.AssignmentExpression,
                    operator: operator,
                    left: left,
                    right: right
                  };
                },
                createBinaryExpression: function (operator, left, right) {
                  var type = operator === '||' || operator === '&&' ? Syntax.LogicalExpression : Syntax.BinaryExpression;
                  return {
                    type: type,
                    operator: operator,
                    left: left,
                    right: right
                  };
                },
                createBlockStatement: function (body) {
                  return {
                    type: Syntax.BlockStatement,
                    body: body
                  };
                },
                createBreakStatement: function (label) {
                  return {
                    type: Syntax.BreakStatement,
                    label: label
                  };
                },
                createCallExpression: function (callee, args) {
                  return {
                    type: Syntax.CallExpression,
                    callee: callee,
                    'arguments': args
                  };
                },
                createCatchClause: function (param, body) {
                  return {
                    type: Syntax.CatchClause,
                    param: param,
                    body: body
                  };
                },
                createConditionalExpression: function (test, consequent, alternate) {
                  return {
                    type: Syntax.ConditionalExpression,
                    test: test,
                    consequent: consequent,
                    alternate: alternate
                  };
                },
                createContinueStatement: function (label) {
                  return {
                    type: Syntax.ContinueStatement,
                    label: label
                  };
                },
                createDebuggerStatement: function () {
                  return {
                    type: Syntax.DebuggerStatement
                  };
                },
                createDoWhileStatement: function (body, test) {
                  return {
                    type: Syntax.DoWhileStatement,
                    body: body,
                    test: test
                  };
                },
                createEmptyStatement: function () {
                  return {
                    type: Syntax.EmptyStatement
                  };
                },
                createExpressionStatement: function (expression) {
                  return {
                    type: Syntax.ExpressionStatement,
                    expression: expression
                  };
                },
                createForStatement: function (init, test, update, body) {
                  return {
                    type: Syntax.ForStatement,
                    init: init,
                    test: test,
                    update: update,
                    body: body
                  };
                },
                createForInStatement: function (left, right, body) {
                  return {
                    type: Syntax.ForInStatement,
                    left: left,
                    right: right,
                    body: body,
                    each: false
                  };
                },
                createFunctionDeclaration: function (id, params, defaults, body) {
                  return {
                    type: Syntax.FunctionDeclaration,
                    id: id,
                    params: params,
                    defaults: defaults,
                    body: body,
                    rest: null,
                    generator: false,
                    expression: false
                  };
                },
                createFunctionExpression: function (id, params, defaults, body) {
                  return {
                    type: Syntax.FunctionExpression,
                    id: id,
                    params: params,
                    defaults: defaults,
                    body: body,
                    rest: null,
                    generator: false,
                    expression: false
                  };
                },
                createIdentifier: function (name) {
                  return {
                    type: Syntax.Identifier,
                    name: name
                  };
                },
                createIfStatement: function (test, consequent, alternate) {
                  return {
                    type: Syntax.IfStatement,
                    test: test,
                    consequent: consequent,
                    alternate: alternate
                  };
                },
                createLabeledStatement: function (label, body) {
                  return {
                    type: Syntax.LabeledStatement,
                    label: label,
                    body: body
                  };
                },
                createLiteral: function (token) {
                  return {
                    type: Syntax.Literal,
                    value: token.value,
                    raw: source.slice(token.start, token.end)
                  };
                },
                createMemberExpression: function (accessor, object, property) {
                  return {
                    type: Syntax.MemberExpression,
                    computed: accessor === '[',
                    object: object,
                    property: property
                  };
                },
                createNewExpression: function (callee, args) {
                  return {
                    type: Syntax.NewExpression,
                    callee: callee,
                    'arguments': args
                  };
                },
                createObjectExpression: function (properties) {
                  return {
                    type: Syntax.ObjectExpression,
                    properties: properties
                  };
                },
                createPostfixExpression: function (operator, argument) {
                  return {
                    type: Syntax.UpdateExpression,
                    operator: operator,
                    argument: argument,
                    prefix: false
                  };
                },
                createProgram: function (body) {
                  return {
                    type: Syntax.Program,
                    body: body
                  };
                },
                createProperty: function (kind, key, value) {
                  return {
                    type: Syntax.Property,
                    key: key,
                    value: value,
                    kind: kind
                  };
                },
                createReturnStatement: function (argument) {
                  return {
                    type: Syntax.ReturnStatement,
                    argument: argument
                  };
                },
                createSequenceExpression: function (expressions) {
                  return {
                    type: Syntax.SequenceExpression,
                    expressions: expressions
                  };
                },
                createSwitchCase: function (test, consequent) {
                  return {
                    type: Syntax.SwitchCase,
                    test: test,
                    consequent: consequent
                  };
                },
                createSwitchStatement: function (discriminant, cases) {
                  return {
                    type: Syntax.SwitchStatement,
                    discriminant: discriminant,
                    cases: cases
                  };
                },
                createThisExpression: function () {
                  return {
                    type: Syntax.ThisExpression
                  };
                },
                createThrowStatement: function (argument) {
                  return {
                    type: Syntax.ThrowStatement,
                    argument: argument
                  };
                },
                createTryStatement: function (block, guardedHandlers, handlers, finalizer) {
                  return {
                    type: Syntax.TryStatement,
                    block: block,
                    guardedHandlers: guardedHandlers,
                    handlers: handlers,
                    finalizer: finalizer
                  };
                },
                createUnaryExpression: function (operator, argument) {
                  if (operator === '++' || operator === '--') {
                    return {
                      type: Syntax.UpdateExpression,
                      operator: operator,
                      argument: argument,
                      prefix: true
                    };
                  }
                  return {
                    type: Syntax.UnaryExpression,
                    operator: operator,
                    argument: argument,
                    prefix: true
                  };
                },
                createVariableDeclaration: function (declarations, kind) {
                  return {
                    type: Syntax.VariableDeclaration,
                    declarations: declarations,
                    kind: kind
                  };
                },
                createVariableDeclarator: function (id, init) {
                  return {
                    type: Syntax.VariableDeclarator,
                    id: id,
                    init: init
                  };
                },
                createWhileStatement: function (test, body) {
                  return {
                    type: Syntax.WhileStatement,
                    test: test,
                    body: body
                  };
                },
                createWithStatement: function (object, body) {
                  return {
                    type: Syntax.WithStatement,
                    object: object,
                    body: body
                  };
                }
              };

              // Return true if there is a line terminator before the next token.

              function peekLineTerminator() {
                var pos, line, start, found;
                pos = index;
                line = lineNumber;
                start = lineStart;
                skipComment();
                found = lineNumber !== line;
                index = pos;
                lineNumber = line;
                lineStart = start;
                return found;
              }

              // Throw an exception

              function throwError(token, messageFormat) {
                var error,
                  args = Array.prototype.slice.call(arguments, 2),
                  msg = messageFormat.replace(/%(\d)/g, function (whole, index) {
                    assert(index < args.length, 'Message reference must be in range');
                    return args[index];
                  });
                if (typeof token.lineNumber === 'number') {
                  error = new Error('Line ' + token.lineNumber + ': ' + msg);
                  error.index = token.start;
                  error.lineNumber = token.lineNumber;
                  error.column = token.start - lineStart + 1;
                } else {
                  error = new Error('Line ' + lineNumber + ': ' + msg);
                  error.index = index;
                  error.lineNumber = lineNumber;
                  error.column = index - lineStart + 1;
                }
                error.description = msg;
                throw error;
              }
              function throwErrorTolerant() {
                try {
                  throwError.apply(null, arguments);
                } catch (e) {
                  if (extra.errors) {
                    extra.errors.push(e);
                  } else {
                    throw e;
                  }
                }
              }

              // Throw an exception because of the token.

              function throwUnexpected(token) {
                if (token.type === Token.EOF) {
                  throwError(token, Messages.UnexpectedEOS);
                }
                if (token.type === Token.NumericLiteral) {
                  throwError(token, Messages.UnexpectedNumber);
                }
                if (token.type === Token.StringLiteral) {
                  throwError(token, Messages.UnexpectedString);
                }
                if (token.type === Token.Identifier) {
                  throwError(token, Messages.UnexpectedIdentifier);
                }
                if (token.type === Token.Keyword) {
                  if (isFutureReservedWord(token.value)) {
                    throwError(token, Messages.UnexpectedReserved);
                  } else if (strict && isStrictModeReservedWord(token.value)) {
                    throwErrorTolerant(token, Messages.StrictReservedWord);
                    return;
                  }
                  throwError(token, Messages.UnexpectedToken, token.value);
                }

                // BooleanLiteral, NullLiteral, or Punctuator.
                throwError(token, Messages.UnexpectedToken, token.value);
              }

              // Expect the next token to match the specified punctuator.
              // If not, an exception will be thrown.

              function expect(value) {
                var token = lex();
                if (token.type !== Token.Punctuator || token.value !== value) {
                  throwUnexpected(token);
                }
              }

              // Expect the next token to match the specified keyword.
              // If not, an exception will be thrown.

              function expectKeyword(keyword) {
                var token = lex();
                if (token.type !== Token.Keyword || token.value !== keyword) {
                  throwUnexpected(token);
                }
              }

              // Return true if the next token matches the specified punctuator.

              function match(value) {
                return lookahead.type === Token.Punctuator && lookahead.value === value;
              }

              // Return true if the next token matches the specified keyword

              function matchKeyword(keyword) {
                return lookahead.type === Token.Keyword && lookahead.value === keyword;
              }

              // Return true if the next token is an assignment operator

              function matchAssign() {
                var op;
                if (lookahead.type !== Token.Punctuator) {
                  return false;
                }
                op = lookahead.value;
                return op === '=' || op === '*=' || op === '/=' || op === '%=' || op === '+=' || op === '-=' || op === '<<=' || op === '>>=' || op === '>>>=' || op === '&=' || op === '^=' || op === '|=';
              }
              function consumeSemicolon() {
                var line;

                // Catch the very common case first: immediately a semicolon (U+003B).
                if (source.charCodeAt(index) === 0x3B || match(';')) {
                  lex();
                  return;
                }
                line = lineNumber;
                skipComment();
                if (lineNumber !== line) {
                  return;
                }
                if (lookahead.type !== Token.EOF && !match('}')) {
                  throwUnexpected(lookahead);
                }
              }

              // Return true if provided expression is LeftHandSideExpression

              function isLeftHandSide(expr) {
                return expr.type === Syntax.Identifier || expr.type === Syntax.MemberExpression;
              }

              // 11.1.4 Array Initialiser

              function parseArrayInitialiser() {
                var elements = [],
                  startToken;
                startToken = lookahead;
                expect('[');
                while (!match(']')) {
                  if (match(',')) {
                    lex();
                    elements.push(null);
                  } else {
                    elements.push(parseAssignmentExpression());
                    if (!match(']')) {
                      expect(',');
                    }
                  }
                }
                lex();
                return delegate.markEnd(delegate.createArrayExpression(elements), startToken);
              }

              // 11.1.5 Object Initialiser

              function parsePropertyFunction(param, first) {
                var previousStrict, body, startToken;
                previousStrict = strict;
                startToken = lookahead;
                body = parseFunctionSourceElements();
                if (first && strict && isRestrictedWord(param[0].name)) {
                  throwErrorTolerant(first, Messages.StrictParamName);
                }
                strict = previousStrict;
                return delegate.markEnd(delegate.createFunctionExpression(null, param, [], body), startToken);
              }
              function parseObjectPropertyKey() {
                var token, startToken;
                startToken = lookahead;
                token = lex();

                // Note: This function is called only from parseObjectProperty(), where
                // EOF and Punctuator tokens are already filtered out.

                if (token.type === Token.StringLiteral || token.type === Token.NumericLiteral) {
                  if (strict && token.octal) {
                    throwErrorTolerant(token, Messages.StrictOctalLiteral);
                  }
                  return delegate.markEnd(delegate.createLiteral(token), startToken);
                }
                return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
              }
              function parseObjectProperty() {
                var token, key, id, value, param, startToken;
                token = lookahead;
                startToken = lookahead;
                if (token.type === Token.Identifier) {
                  id = parseObjectPropertyKey();

                  // Property Assignment: Getter and Setter.

                  if (token.value === 'get' && !match(':')) {
                    key = parseObjectPropertyKey();
                    expect('(');
                    expect(')');
                    value = parsePropertyFunction([]);
                    return delegate.markEnd(delegate.createProperty('get', key, value), startToken);
                  }
                  if (token.value === 'set' && !match(':')) {
                    key = parseObjectPropertyKey();
                    expect('(');
                    token = lookahead;
                    if (token.type !== Token.Identifier) {
                      expect(')');
                      throwErrorTolerant(token, Messages.UnexpectedToken, token.value);
                      value = parsePropertyFunction([]);
                    } else {
                      param = [parseVariableIdentifier()];
                      expect(')');
                      value = parsePropertyFunction(param, token);
                    }
                    return delegate.markEnd(delegate.createProperty('set', key, value), startToken);
                  }
                  expect(':');
                  value = parseAssignmentExpression();
                  return delegate.markEnd(delegate.createProperty('init', id, value), startToken);
                }
                if (token.type === Token.EOF || token.type === Token.Punctuator) {
                  throwUnexpected(token);
                } else {
                  key = parseObjectPropertyKey();
                  expect(':');
                  value = parseAssignmentExpression();
                  return delegate.markEnd(delegate.createProperty('init', key, value), startToken);
                }
              }
              function parseObjectInitialiser() {
                var properties = [],
                  property,
                  name,
                  key,
                  kind,
                  map = {},
                  toString = String,
                  startToken;
                startToken = lookahead;
                expect('{');
                while (!match('}')) {
                  property = parseObjectProperty();
                  if (property.key.type === Syntax.Identifier) {
                    name = property.key.name;
                  } else {
                    name = toString(property.key.value);
                  }
                  kind = property.kind === 'init' ? PropertyKind.Data : property.kind === 'get' ? PropertyKind.Get : PropertyKind.Set;
                  key = '$' + name;
                  if (Object.prototype.hasOwnProperty.call(map, key)) {
                    if (map[key] === PropertyKind.Data) {
                      if (strict && kind === PropertyKind.Data) {
                        throwErrorTolerant({}, Messages.StrictDuplicateProperty);
                      } else if (kind !== PropertyKind.Data) {
                        throwErrorTolerant({}, Messages.AccessorDataProperty);
                      }
                    } else {
                      if (kind === PropertyKind.Data) {
                        throwErrorTolerant({}, Messages.AccessorDataProperty);
                      } else if (map[key] & kind) {
                        throwErrorTolerant({}, Messages.AccessorGetSet);
                      }
                    }
                    map[key] |= kind;
                  } else {
                    map[key] = kind;
                  }
                  properties.push(property);
                  if (!match('}')) {
                    expect(',');
                  }
                }
                expect('}');
                return delegate.markEnd(delegate.createObjectExpression(properties), startToken);
              }

              // 11.1.6 The Grouping Operator

              function parseGroupExpression() {
                var expr;
                expect('(');
                expr = parseExpression();
                expect(')');
                return expr;
              }

              // 11.1 Primary Expressions

              function parsePrimaryExpression() {
                var type, token, expr, startToken;
                if (match('(')) {
                  return parseGroupExpression();
                }
                if (match('[')) {
                  return parseArrayInitialiser();
                }
                if (match('{')) {
                  return parseObjectInitialiser();
                }
                type = lookahead.type;
                startToken = lookahead;
                if (type === Token.Identifier) {
                  expr = delegate.createIdentifier(lex().value);
                } else if (type === Token.StringLiteral || type === Token.NumericLiteral) {
                  if (strict && lookahead.octal) {
                    throwErrorTolerant(lookahead, Messages.StrictOctalLiteral);
                  }
                  expr = delegate.createLiteral(lex());
                } else if (type === Token.Keyword) {
                  if (matchKeyword('function')) {
                    return parseFunctionExpression();
                  }
                  if (matchKeyword('this')) {
                    lex();
                    expr = delegate.createThisExpression();
                  } else {
                    throwUnexpected(lex());
                  }
                } else if (type === Token.BooleanLiteral) {
                  token = lex();
                  token.value = token.value === 'true';
                  expr = delegate.createLiteral(token);
                } else if (type === Token.NullLiteral) {
                  token = lex();
                  token.value = null;
                  expr = delegate.createLiteral(token);
                } else if (match('/') || match('/=')) {
                  if (typeof extra.tokens !== 'undefined') {
                    expr = delegate.createLiteral(collectRegex());
                  } else {
                    expr = delegate.createLiteral(scanRegExp());
                  }
                  peek();
                } else {
                  throwUnexpected(lex());
                }
                return delegate.markEnd(expr, startToken);
              }

              // 11.2 Left-Hand-Side Expressions

              function parseArguments() {
                var args = [];
                expect('(');
                if (!match(')')) {
                  while (index < length) {
                    args.push(parseAssignmentExpression());
                    if (match(')')) {
                      break;
                    }
                    expect(',');
                  }
                }
                expect(')');
                return args;
              }
              function parseNonComputedProperty() {
                var token, startToken;
                startToken = lookahead;
                token = lex();
                if (!isIdentifierName(token)) {
                  throwUnexpected(token);
                }
                return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
              }
              function parseNonComputedMember() {
                expect('.');
                return parseNonComputedProperty();
              }
              function parseComputedMember() {
                var expr;
                expect('[');
                expr = parseExpression();
                expect(']');
                return expr;
              }
              function parseNewExpression() {
                var callee, args, startToken;
                startToken = lookahead;
                expectKeyword('new');
                callee = parseLeftHandSideExpression();
                args = match('(') ? parseArguments() : [];
                return delegate.markEnd(delegate.createNewExpression(callee, args), startToken);
              }
              function parseLeftHandSideExpressionAllowCall() {
                var previousAllowIn, expr, args, property, startToken;
                startToken = lookahead;
                previousAllowIn = state.allowIn;
                state.allowIn = true;
                expr = matchKeyword('new') ? parseNewExpression() : parsePrimaryExpression();
                state.allowIn = previousAllowIn;
                for (;;) {
                  if (match('.')) {
                    property = parseNonComputedMember();
                    expr = delegate.createMemberExpression('.', expr, property);
                  } else if (match('(')) {
                    args = parseArguments();
                    expr = delegate.createCallExpression(expr, args);
                  } else if (match('[')) {
                    property = parseComputedMember();
                    expr = delegate.createMemberExpression('[', expr, property);
                  } else {
                    break;
                  }
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }
              function parseLeftHandSideExpression() {
                var previousAllowIn, expr, property, startToken;
                startToken = lookahead;
                previousAllowIn = state.allowIn;
                expr = matchKeyword('new') ? parseNewExpression() : parsePrimaryExpression();
                state.allowIn = previousAllowIn;
                while (match('.') || match('[')) {
                  if (match('[')) {
                    property = parseComputedMember();
                    expr = delegate.createMemberExpression('[', expr, property);
                  } else {
                    property = parseNonComputedMember();
                    expr = delegate.createMemberExpression('.', expr, property);
                  }
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }

              // 11.3 Postfix Expressions

              function parsePostfixExpression() {
                var expr,
                  token,
                  startToken = lookahead;
                expr = parseLeftHandSideExpressionAllowCall();
                if (lookahead.type === Token.Punctuator) {
                  if ((match('++') || match('--')) && !peekLineTerminator()) {
                    // 11.3.1, 11.3.2
                    if (strict && expr.type === Syntax.Identifier && isRestrictedWord(expr.name)) {
                      throwErrorTolerant({}, Messages.StrictLHSPostfix);
                    }
                    if (!isLeftHandSide(expr)) {
                      throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
                    }
                    token = lex();
                    expr = delegate.markEnd(delegate.createPostfixExpression(token.value, expr), startToken);
                  }
                }
                return expr;
              }

              // 11.4 Unary Operators

              function parseUnaryExpression() {
                var token, expr, startToken;
                if (lookahead.type !== Token.Punctuator && lookahead.type !== Token.Keyword) {
                  expr = parsePostfixExpression();
                } else if (match('++') || match('--')) {
                  startToken = lookahead;
                  token = lex();
                  expr = parseUnaryExpression();
                  // 11.4.4, 11.4.5
                  if (strict && expr.type === Syntax.Identifier && isRestrictedWord(expr.name)) {
                    throwErrorTolerant({}, Messages.StrictLHSPrefix);
                  }
                  if (!isLeftHandSide(expr)) {
                    throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
                  }
                  expr = delegate.createUnaryExpression(token.value, expr);
                  expr = delegate.markEnd(expr, startToken);
                } else if (match('+') || match('-') || match('~') || match('!')) {
                  startToken = lookahead;
                  token = lex();
                  expr = parseUnaryExpression();
                  expr = delegate.createUnaryExpression(token.value, expr);
                  expr = delegate.markEnd(expr, startToken);
                } else if (matchKeyword('delete') || matchKeyword('void') || matchKeyword('typeof')) {
                  startToken = lookahead;
                  token = lex();
                  expr = parseUnaryExpression();
                  expr = delegate.createUnaryExpression(token.value, expr);
                  expr = delegate.markEnd(expr, startToken);
                  if (strict && expr.operator === 'delete' && expr.argument.type === Syntax.Identifier) {
                    throwErrorTolerant({}, Messages.StrictDelete);
                  }
                } else {
                  expr = parsePostfixExpression();
                }
                return expr;
              }
              function binaryPrecedence(token, allowIn) {
                var prec = 0;
                if (token.type !== Token.Punctuator && token.type !== Token.Keyword) {
                  return 0;
                }
                switch (token.value) {
                  case '||':
                    prec = 1;
                    break;
                  case '&&':
                    prec = 2;
                    break;
                  case '|':
                    prec = 3;
                    break;
                  case '^':
                    prec = 4;
                    break;
                  case '&':
                    prec = 5;
                    break;
                  case '==':
                  case '!=':
                  case '===':
                  case '!==':
                    prec = 6;
                    break;
                  case '<':
                  case '>':
                  case '<=':
                  case '>=':
                  case 'instanceof':
                    prec = 7;
                    break;
                  case 'in':
                    prec = allowIn ? 7 : 0;
                    break;
                  case '<<':
                  case '>>':
                  case '>>>':
                    prec = 8;
                    break;
                  case '+':
                  case '-':
                    prec = 9;
                    break;
                  case '*':
                  case '/':
                  case '%':
                    prec = 11;
                    break;
                }
                return prec;
              }

              // 11.5 Multiplicative Operators
              // 11.6 Additive Operators
              // 11.7 Bitwise Shift Operators
              // 11.8 Relational Operators
              // 11.9 Equality Operators
              // 11.10 Binary Bitwise Operators
              // 11.11 Binary Logical Operators

              function parseBinaryExpression() {
                var marker, markers, expr, token, prec, stack, right, operator, left, i;
                marker = lookahead;
                left = parseUnaryExpression();
                token = lookahead;
                prec = binaryPrecedence(token, state.allowIn);
                if (prec === 0) {
                  return left;
                }
                token.prec = prec;
                lex();
                markers = [marker, lookahead];
                right = parseUnaryExpression();
                stack = [left, token, right];
                while ((prec = binaryPrecedence(lookahead, state.allowIn)) > 0) {
                  // Reduce: make a binary expression from the three topmost entries.
                  while (stack.length > 2 && prec <= stack[stack.length - 2].prec) {
                    right = stack.pop();
                    operator = stack.pop().value;
                    left = stack.pop();
                    expr = delegate.createBinaryExpression(operator, left, right);
                    markers.pop();
                    marker = markers[markers.length - 1];
                    delegate.markEnd(expr, marker);
                    stack.push(expr);
                  }

                  // Shift.
                  token = lex();
                  token.prec = prec;
                  stack.push(token);
                  markers.push(lookahead);
                  expr = parseUnaryExpression();
                  stack.push(expr);
                }

                // Final reduce to clean-up the stack.
                i = stack.length - 1;
                expr = stack[i];
                markers.pop();
                while (i > 1) {
                  expr = delegate.createBinaryExpression(stack[i - 1].value, stack[i - 2], expr);
                  i -= 2;
                  marker = markers.pop();
                  delegate.markEnd(expr, marker);
                }
                return expr;
              }

              // 11.12 Conditional Operator

              function parseConditionalExpression() {
                var expr, previousAllowIn, consequent, alternate, startToken;
                startToken = lookahead;
                expr = parseBinaryExpression();
                if (match('?')) {
                  lex();
                  previousAllowIn = state.allowIn;
                  state.allowIn = true;
                  consequent = parseAssignmentExpression();
                  state.allowIn = previousAllowIn;
                  expect(':');
                  alternate = parseAssignmentExpression();
                  expr = delegate.createConditionalExpression(expr, consequent, alternate);
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }

              // 11.13 Assignment Operators

              function parseAssignmentExpression() {
                var token, left, right, node, startToken;
                token = lookahead;
                startToken = lookahead;
                node = left = parseConditionalExpression();
                if (matchAssign()) {
                  // LeftHandSideExpression
                  if (!isLeftHandSide(left)) {
                    throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
                  }

                  // 11.13.1
                  if (strict && left.type === Syntax.Identifier && isRestrictedWord(left.name)) {
                    throwErrorTolerant(token, Messages.StrictLHSAssignment);
                  }
                  token = lex();
                  right = parseAssignmentExpression();
                  node = delegate.markEnd(delegate.createAssignmentExpression(token.value, left, right), startToken);
                }
                return node;
              }

              // 11.14 Comma Operator

              function parseExpression() {
                var expr,
                  startToken = lookahead;
                expr = parseAssignmentExpression();
                if (match(',')) {
                  expr = delegate.createSequenceExpression([expr]);
                  while (index < length) {
                    if (!match(',')) {
                      break;
                    }
                    lex();
                    expr.expressions.push(parseAssignmentExpression());
                  }
                  delegate.markEnd(expr, startToken);
                }
                return expr;
              }

              // 12.1 Block

              function parseStatementList() {
                var list = [],
                  statement;
                while (index < length) {
                  if (match('}')) {
                    break;
                  }
                  statement = parseSourceElement();
                  if (typeof statement === 'undefined') {
                    break;
                  }
                  list.push(statement);
                }
                return list;
              }
              function parseBlock() {
                var block, startToken;
                startToken = lookahead;
                expect('{');
                block = parseStatementList();
                expect('}');
                return delegate.markEnd(delegate.createBlockStatement(block), startToken);
              }

              // 12.2 Variable Statement

              function parseVariableIdentifier() {
                var token, startToken;
                startToken = lookahead;
                token = lex();
                if (token.type !== Token.Identifier) {
                  throwUnexpected(token);
                }
                return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
              }
              function parseVariableDeclaration(kind) {
                var init = null,
                  id,
                  startToken;
                startToken = lookahead;
                id = parseVariableIdentifier();

                // 12.2.1
                if (strict && isRestrictedWord(id.name)) {
                  throwErrorTolerant({}, Messages.StrictVarName);
                }
                if (kind === 'const') {
                  expect('=');
                  init = parseAssignmentExpression();
                } else if (match('=')) {
                  lex();
                  init = parseAssignmentExpression();
                }
                return delegate.markEnd(delegate.createVariableDeclarator(id, init), startToken);
              }
              function parseVariableDeclarationList(kind) {
                var list = [];
                do {
                  list.push(parseVariableDeclaration(kind));
                  if (!match(',')) {
                    break;
                  }
                  lex();
                } while (index < length);
                return list;
              }
              function parseVariableStatement() {
                var declarations;
                expectKeyword('var');
                declarations = parseVariableDeclarationList();
                consumeSemicolon();
                return delegate.createVariableDeclaration(declarations, 'var');
              }

              // kind may be `const` or `let`
              // Both are experimental and not in the specification yet.
              // see http://wiki.ecmascript.org/doku.php?id=harmony:const
              // and http://wiki.ecmascript.org/doku.php?id=harmony:let
              function parseConstLetDeclaration(kind) {
                var declarations, startToken;
                startToken = lookahead;
                expectKeyword(kind);
                declarations = parseVariableDeclarationList(kind);
                consumeSemicolon();
                return delegate.markEnd(delegate.createVariableDeclaration(declarations, kind), startToken);
              }

              // 12.3 Empty Statement

              function parseEmptyStatement() {
                expect(';');
                return delegate.createEmptyStatement();
              }

              // 12.4 Expression Statement

              function parseExpressionStatement() {
                var expr = parseExpression();
                consumeSemicolon();
                return delegate.createExpressionStatement(expr);
              }

              // 12.5 If statement

              function parseIfStatement() {
                var test, consequent, alternate;
                expectKeyword('if');
                expect('(');
                test = parseExpression();
                expect(')');
                consequent = parseStatement();
                if (matchKeyword('else')) {
                  lex();
                  alternate = parseStatement();
                } else {
                  alternate = null;
                }
                return delegate.createIfStatement(test, consequent, alternate);
              }

              // 12.6 Iteration Statements

              function parseDoWhileStatement() {
                var body, test, oldInIteration;
                expectKeyword('do');
                oldInIteration = state.inIteration;
                state.inIteration = true;
                body = parseStatement();
                state.inIteration = oldInIteration;
                expectKeyword('while');
                expect('(');
                test = parseExpression();
                expect(')');
                if (match(';')) {
                  lex();
                }
                return delegate.createDoWhileStatement(body, test);
              }
              function parseWhileStatement() {
                var test, body, oldInIteration;
                expectKeyword('while');
                expect('(');
                test = parseExpression();
                expect(')');
                oldInIteration = state.inIteration;
                state.inIteration = true;
                body = parseStatement();
                state.inIteration = oldInIteration;
                return delegate.createWhileStatement(test, body);
              }
              function parseForVariableDeclaration() {
                var token, declarations, startToken;
                startToken = lookahead;
                token = lex();
                declarations = parseVariableDeclarationList();
                return delegate.markEnd(delegate.createVariableDeclaration(declarations, token.value), startToken);
              }
              function parseForStatement() {
                var init, test, update, left, right, body, oldInIteration;
                init = test = update = null;
                expectKeyword('for');
                expect('(');
                if (match(';')) {
                  lex();
                } else {
                  if (matchKeyword('var') || matchKeyword('let')) {
                    state.allowIn = false;
                    init = parseForVariableDeclaration();
                    state.allowIn = true;
                    if (init.declarations.length === 1 && matchKeyword('in')) {
                      lex();
                      left = init;
                      right = parseExpression();
                      init = null;
                    }
                  } else {
                    state.allowIn = false;
                    init = parseExpression();
                    state.allowIn = true;
                    if (matchKeyword('in')) {
                      // LeftHandSideExpression
                      if (!isLeftHandSide(init)) {
                        throwErrorTolerant({}, Messages.InvalidLHSInForIn);
                      }
                      lex();
                      left = init;
                      right = parseExpression();
                      init = null;
                    }
                  }
                  if (typeof left === 'undefined') {
                    expect(';');
                  }
                }
                if (typeof left === 'undefined') {
                  if (!match(';')) {
                    test = parseExpression();
                  }
                  expect(';');
                  if (!match(')')) {
                    update = parseExpression();
                  }
                }
                expect(')');
                oldInIteration = state.inIteration;
                state.inIteration = true;
                body = parseStatement();
                state.inIteration = oldInIteration;
                return typeof left === 'undefined' ? delegate.createForStatement(init, test, update, body) : delegate.createForInStatement(left, right, body);
              }

              // 12.7 The continue statement

              function parseContinueStatement() {
                var label = null,
                  key;
                expectKeyword('continue');

                // Optimize the most common form: 'continue;'.
                if (source.charCodeAt(index) === 0x3B) {
                  lex();
                  if (!state.inIteration) {
                    throwError({}, Messages.IllegalContinue);
                  }
                  return delegate.createContinueStatement(null);
                }
                if (peekLineTerminator()) {
                  if (!state.inIteration) {
                    throwError({}, Messages.IllegalContinue);
                  }
                  return delegate.createContinueStatement(null);
                }
                if (lookahead.type === Token.Identifier) {
                  label = parseVariableIdentifier();
                  key = '$' + label.name;
                  if (!Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
                    throwError({}, Messages.UnknownLabel, label.name);
                  }
                }
                consumeSemicolon();
                if (label === null && !state.inIteration) {
                  throwError({}, Messages.IllegalContinue);
                }
                return delegate.createContinueStatement(label);
              }

              // 12.8 The break statement

              function parseBreakStatement() {
                var label = null,
                  key;
                expectKeyword('break');

                // Catch the very common case first: immediately a semicolon (U+003B).
                if (source.charCodeAt(index) === 0x3B) {
                  lex();
                  if (!(state.inIteration || state.inSwitch)) {
                    throwError({}, Messages.IllegalBreak);
                  }
                  return delegate.createBreakStatement(null);
                }
                if (peekLineTerminator()) {
                  if (!(state.inIteration || state.inSwitch)) {
                    throwError({}, Messages.IllegalBreak);
                  }
                  return delegate.createBreakStatement(null);
                }
                if (lookahead.type === Token.Identifier) {
                  label = parseVariableIdentifier();
                  key = '$' + label.name;
                  if (!Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
                    throwError({}, Messages.UnknownLabel, label.name);
                  }
                }
                consumeSemicolon();
                if (label === null && !(state.inIteration || state.inSwitch)) {
                  throwError({}, Messages.IllegalBreak);
                }
                return delegate.createBreakStatement(label);
              }

              // 12.9 The return statement

              function parseReturnStatement() {
                var argument = null;
                expectKeyword('return');
                if (!state.inFunctionBody) {
                  throwErrorTolerant({}, Messages.IllegalReturn);
                }

                // 'return' followed by a space and an identifier is very common.
                if (source.charCodeAt(index) === 0x20) {
                  if (isIdentifierStart(source.charCodeAt(index + 1))) {
                    argument = parseExpression();
                    consumeSemicolon();
                    return delegate.createReturnStatement(argument);
                  }
                }
                if (peekLineTerminator()) {
                  return delegate.createReturnStatement(null);
                }
                if (!match(';')) {
                  if (!match('}') && lookahead.type !== Token.EOF) {
                    argument = parseExpression();
                  }
                }
                consumeSemicolon();
                return delegate.createReturnStatement(argument);
              }

              // 12.10 The with statement

              function parseWithStatement() {
                var object, body;
                if (strict) {
                  // TODO(ikarienator): Should we update the test cases instead?
                  skipComment();
                  throwErrorTolerant({}, Messages.StrictModeWith);
                }
                expectKeyword('with');
                expect('(');
                object = parseExpression();
                expect(')');
                body = parseStatement();
                return delegate.createWithStatement(object, body);
              }

              // 12.10 The swith statement

              function parseSwitchCase() {
                var test,
                  consequent = [],
                  statement,
                  startToken;
                startToken = lookahead;
                if (matchKeyword('default')) {
                  lex();
                  test = null;
                } else {
                  expectKeyword('case');
                  test = parseExpression();
                }
                expect(':');
                while (index < length) {
                  if (match('}') || matchKeyword('default') || matchKeyword('case')) {
                    break;
                  }
                  statement = parseStatement();
                  consequent.push(statement);
                }
                return delegate.markEnd(delegate.createSwitchCase(test, consequent), startToken);
              }
              function parseSwitchStatement() {
                var discriminant, cases, clause, oldInSwitch, defaultFound;
                expectKeyword('switch');
                expect('(');
                discriminant = parseExpression();
                expect(')');
                expect('{');
                cases = [];
                if (match('}')) {
                  lex();
                  return delegate.createSwitchStatement(discriminant, cases);
                }
                oldInSwitch = state.inSwitch;
                state.inSwitch = true;
                defaultFound = false;
                while (index < length) {
                  if (match('}')) {
                    break;
                  }
                  clause = parseSwitchCase();
                  if (clause.test === null) {
                    if (defaultFound) {
                      throwError({}, Messages.MultipleDefaultsInSwitch);
                    }
                    defaultFound = true;
                  }
                  cases.push(clause);
                }
                state.inSwitch = oldInSwitch;
                expect('}');
                return delegate.createSwitchStatement(discriminant, cases);
              }

              // 12.13 The throw statement

              function parseThrowStatement() {
                var argument;
                expectKeyword('throw');
                if (peekLineTerminator()) {
                  throwError({}, Messages.NewlineAfterThrow);
                }
                argument = parseExpression();
                consumeSemicolon();
                return delegate.createThrowStatement(argument);
              }

              // 12.14 The try statement

              function parseCatchClause() {
                var param, body, startToken;
                startToken = lookahead;
                expectKeyword('catch');
                expect('(');
                if (match(')')) {
                  throwUnexpected(lookahead);
                }
                param = parseVariableIdentifier();
                // 12.14.1
                if (strict && isRestrictedWord(param.name)) {
                  throwErrorTolerant({}, Messages.StrictCatchVariable);
                }
                expect(')');
                body = parseBlock();
                return delegate.markEnd(delegate.createCatchClause(param, body), startToken);
              }
              function parseTryStatement() {
                var block,
                  handlers = [],
                  finalizer = null;
                expectKeyword('try');
                block = parseBlock();
                if (matchKeyword('catch')) {
                  handlers.push(parseCatchClause());
                }
                if (matchKeyword('finally')) {
                  lex();
                  finalizer = parseBlock();
                }
                if (handlers.length === 0 && !finalizer) {
                  throwError({}, Messages.NoCatchOrFinally);
                }
                return delegate.createTryStatement(block, [], handlers, finalizer);
              }

              // 12.15 The debugger statement

              function parseDebuggerStatement() {
                expectKeyword('debugger');
                consumeSemicolon();
                return delegate.createDebuggerStatement();
              }

              // 12 Statements

              function parseStatement() {
                var type = lookahead.type,
                  expr,
                  labeledBody,
                  key,
                  startToken;
                if (type === Token.EOF) {
                  throwUnexpected(lookahead);
                }
                if (type === Token.Punctuator && lookahead.value === '{') {
                  return parseBlock();
                }
                startToken = lookahead;
                if (type === Token.Punctuator) {
                  switch (lookahead.value) {
                    case ';':
                      return delegate.markEnd(parseEmptyStatement(), startToken);
                    case '(':
                      return delegate.markEnd(parseExpressionStatement(), startToken);
                  }
                }
                if (type === Token.Keyword) {
                  switch (lookahead.value) {
                    case 'break':
                      return delegate.markEnd(parseBreakStatement(), startToken);
                    case 'continue':
                      return delegate.markEnd(parseContinueStatement(), startToken);
                    case 'debugger':
                      return delegate.markEnd(parseDebuggerStatement(), startToken);
                    case 'do':
                      return delegate.markEnd(parseDoWhileStatement(), startToken);
                    case 'for':
                      return delegate.markEnd(parseForStatement(), startToken);
                    case 'function':
                      return delegate.markEnd(parseFunctionDeclaration(), startToken);
                    case 'if':
                      return delegate.markEnd(parseIfStatement(), startToken);
                    case 'return':
                      return delegate.markEnd(parseReturnStatement(), startToken);
                    case 'switch':
                      return delegate.markEnd(parseSwitchStatement(), startToken);
                    case 'throw':
                      return delegate.markEnd(parseThrowStatement(), startToken);
                    case 'try':
                      return delegate.markEnd(parseTryStatement(), startToken);
                    case 'var':
                      return delegate.markEnd(parseVariableStatement(), startToken);
                    case 'while':
                      return delegate.markEnd(parseWhileStatement(), startToken);
                    case 'with':
                      return delegate.markEnd(parseWithStatement(), startToken);
                  }
                }
                expr = parseExpression();

                // 12.12 Labelled Statements
                if (expr.type === Syntax.Identifier && match(':')) {
                  lex();
                  key = '$' + expr.name;
                  if (Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
                    throwError({}, Messages.Redeclaration, 'Label', expr.name);
                  }
                  state.labelSet[key] = true;
                  labeledBody = parseStatement();
                  delete state.labelSet[key];
                  return delegate.markEnd(delegate.createLabeledStatement(expr, labeledBody), startToken);
                }
                consumeSemicolon();
                return delegate.markEnd(delegate.createExpressionStatement(expr), startToken);
              }

              // 13 Function Definition

              function parseFunctionSourceElements() {
                var sourceElement,
                  sourceElements = [],
                  token,
                  directive,
                  firstRestricted,
                  oldLabelSet,
                  oldInIteration,
                  oldInSwitch,
                  oldInFunctionBody,
                  startToken;
                startToken = lookahead;
                expect('{');
                while (index < length) {
                  if (lookahead.type !== Token.StringLiteral) {
                    break;
                  }
                  token = lookahead;
                  sourceElement = parseSourceElement();
                  sourceElements.push(sourceElement);
                  if (sourceElement.expression.type !== Syntax.Literal) {
                    // this is not directive
                    break;
                  }
                  directive = source.slice(token.start + 1, token.end - 1);
                  if (directive === 'use strict') {
                    strict = true;
                    if (firstRestricted) {
                      throwErrorTolerant(firstRestricted, Messages.StrictOctalLiteral);
                    }
                  } else {
                    if (!firstRestricted && token.octal) {
                      firstRestricted = token;
                    }
                  }
                }
                oldLabelSet = state.labelSet;
                oldInIteration = state.inIteration;
                oldInSwitch = state.inSwitch;
                oldInFunctionBody = state.inFunctionBody;
                state.labelSet = {};
                state.inIteration = false;
                state.inSwitch = false;
                state.inFunctionBody = true;
                while (index < length) {
                  if (match('}')) {
                    break;
                  }
                  sourceElement = parseSourceElement();
                  if (typeof sourceElement === 'undefined') {
                    break;
                  }
                  sourceElements.push(sourceElement);
                }
                expect('}');
                state.labelSet = oldLabelSet;
                state.inIteration = oldInIteration;
                state.inSwitch = oldInSwitch;
                state.inFunctionBody = oldInFunctionBody;
                return delegate.markEnd(delegate.createBlockStatement(sourceElements), startToken);
              }
              function parseParams(firstRestricted) {
                var param,
                  params = [],
                  token,
                  stricted,
                  paramSet,
                  key,
                  message;
                expect('(');
                if (!match(')')) {
                  paramSet = {};
                  while (index < length) {
                    token = lookahead;
                    param = parseVariableIdentifier();
                    key = '$' + token.value;
                    if (strict) {
                      if (isRestrictedWord(token.value)) {
                        stricted = token;
                        message = Messages.StrictParamName;
                      }
                      if (Object.prototype.hasOwnProperty.call(paramSet, key)) {
                        stricted = token;
                        message = Messages.StrictParamDupe;
                      }
                    } else if (!firstRestricted) {
                      if (isRestrictedWord(token.value)) {
                        firstRestricted = token;
                        message = Messages.StrictParamName;
                      } else if (isStrictModeReservedWord(token.value)) {
                        firstRestricted = token;
                        message = Messages.StrictReservedWord;
                      } else if (Object.prototype.hasOwnProperty.call(paramSet, key)) {
                        firstRestricted = token;
                        message = Messages.StrictParamDupe;
                      }
                    }
                    params.push(param);
                    paramSet[key] = true;
                    if (match(')')) {
                      break;
                    }
                    expect(',');
                  }
                }
                expect(')');
                return {
                  params: params,
                  stricted: stricted,
                  firstRestricted: firstRestricted,
                  message: message
                };
              }
              function parseFunctionDeclaration() {
                var id,
                  params = [],
                  body,
                  token,
                  stricted,
                  tmp,
                  firstRestricted,
                  message,
                  previousStrict,
                  startToken;
                startToken = lookahead;
                expectKeyword('function');
                token = lookahead;
                id = parseVariableIdentifier();
                if (strict) {
                  if (isRestrictedWord(token.value)) {
                    throwErrorTolerant(token, Messages.StrictFunctionName);
                  }
                } else {
                  if (isRestrictedWord(token.value)) {
                    firstRestricted = token;
                    message = Messages.StrictFunctionName;
                  } else if (isStrictModeReservedWord(token.value)) {
                    firstRestricted = token;
                    message = Messages.StrictReservedWord;
                  }
                }
                tmp = parseParams(firstRestricted);
                params = tmp.params;
                stricted = tmp.stricted;
                firstRestricted = tmp.firstRestricted;
                if (tmp.message) {
                  message = tmp.message;
                }
                previousStrict = strict;
                body = parseFunctionSourceElements();
                if (strict && firstRestricted) {
                  throwError(firstRestricted, message);
                }
                if (strict && stricted) {
                  throwErrorTolerant(stricted, message);
                }
                strict = previousStrict;
                return delegate.markEnd(delegate.createFunctionDeclaration(id, params, [], body), startToken);
              }
              function parseFunctionExpression() {
                var token,
                  id = null,
                  stricted,
                  firstRestricted,
                  message,
                  tmp,
                  params = [],
                  body,
                  previousStrict,
                  startToken;
                startToken = lookahead;
                expectKeyword('function');
                if (!match('(')) {
                  token = lookahead;
                  id = parseVariableIdentifier();
                  if (strict) {
                    if (isRestrictedWord(token.value)) {
                      throwErrorTolerant(token, Messages.StrictFunctionName);
                    }
                  } else {
                    if (isRestrictedWord(token.value)) {
                      firstRestricted = token;
                      message = Messages.StrictFunctionName;
                    } else if (isStrictModeReservedWord(token.value)) {
                      firstRestricted = token;
                      message = Messages.StrictReservedWord;
                    }
                  }
                }
                tmp = parseParams(firstRestricted);
                params = tmp.params;
                stricted = tmp.stricted;
                firstRestricted = tmp.firstRestricted;
                if (tmp.message) {
                  message = tmp.message;
                }
                previousStrict = strict;
                body = parseFunctionSourceElements();
                if (strict && firstRestricted) {
                  throwError(firstRestricted, message);
                }
                if (strict && stricted) {
                  throwErrorTolerant(stricted, message);
                }
                strict = previousStrict;
                return delegate.markEnd(delegate.createFunctionExpression(id, params, [], body), startToken);
              }

              // 14 Program

              function parseSourceElement() {
                if (lookahead.type === Token.Keyword) {
                  switch (lookahead.value) {
                    case 'const':
                    case 'let':
                      return parseConstLetDeclaration(lookahead.value);
                    case 'function':
                      return parseFunctionDeclaration();
                    default:
                      return parseStatement();
                  }
                }
                if (lookahead.type !== Token.EOF) {
                  return parseStatement();
                }
              }
              function parseSourceElements() {
                var sourceElement,
                  sourceElements = [],
                  token,
                  directive,
                  firstRestricted;
                while (index < length) {
                  token = lookahead;
                  if (token.type !== Token.StringLiteral) {
                    break;
                  }
                  sourceElement = parseSourceElement();
                  sourceElements.push(sourceElement);
                  if (sourceElement.expression.type !== Syntax.Literal) {
                    // this is not directive
                    break;
                  }
                  directive = source.slice(token.start + 1, token.end - 1);
                  if (directive === 'use strict') {
                    strict = true;
                    if (firstRestricted) {
                      throwErrorTolerant(firstRestricted, Messages.StrictOctalLiteral);
                    }
                  } else {
                    if (!firstRestricted && token.octal) {
                      firstRestricted = token;
                    }
                  }
                }
                while (index < length) {
                  sourceElement = parseSourceElement();
                  /* istanbul ignore if */
                  if (typeof sourceElement === 'undefined') {
                    break;
                  }
                  sourceElements.push(sourceElement);
                }
                return sourceElements;
              }
              function parseProgram() {
                var body, startToken;
                skipComment();
                peek();
                startToken = lookahead;
                strict = false;
                body = parseSourceElements();
                return delegate.markEnd(delegate.createProgram(body), startToken);
              }
              function filterTokenLocation() {
                var i,
                  entry,
                  token,
                  tokens = [];
                for (i = 0; i < extra.tokens.length; ++i) {
                  entry = extra.tokens[i];
                  token = {
                    type: entry.type,
                    value: entry.value
                  };
                  if (extra.range) {
                    token.range = entry.range;
                  }
                  if (extra.loc) {
                    token.loc = entry.loc;
                  }
                  tokens.push(token);
                }
                extra.tokens = tokens;
              }
              function tokenize(code, options) {
                var toString, token, tokens;
                toString = String;
                if (typeof code !== 'string' && !(code instanceof String)) {
                  code = toString(code);
                }
                delegate = SyntaxTreeDelegate;
                source = code;
                index = 0;
                lineNumber = source.length > 0 ? 1 : 0;
                lineStart = 0;
                length = source.length;
                lookahead = null;
                state = {
                  allowIn: true,
                  labelSet: {},
                  inFunctionBody: false,
                  inIteration: false,
                  inSwitch: false,
                  lastCommentStart: -1
                };
                extra = {};

                // Options matching.
                options = options || {};

                // Of course we collect tokens here.
                options.tokens = true;
                extra.tokens = [];
                extra.tokenize = true;
                // The following two fields are necessary to compute the Regex tokens.
                extra.openParenToken = -1;
                extra.openCurlyToken = -1;
                extra.range = typeof options.range === 'boolean' && options.range;
                extra.loc = typeof options.loc === 'boolean' && options.loc;
                if (typeof options.comment === 'boolean' && options.comment) {
                  extra.comments = [];
                }
                if (typeof options.tolerant === 'boolean' && options.tolerant) {
                  extra.errors = [];
                }
                try {
                  peek();
                  if (lookahead.type === Token.EOF) {
                    return extra.tokens;
                  }
                  token = lex();
                  while (lookahead.type !== Token.EOF) {
                    try {
                      token = lex();
                    } catch (lexError) {
                      token = lookahead;
                      if (extra.errors) {
                        extra.errors.push(lexError);
                        // We have to break on the first error
                        // to avoid infinite loops.
                        break;
                      } else {
                        throw lexError;
                      }
                    }
                  }
                  filterTokenLocation();
                  tokens = extra.tokens;
                  if (typeof extra.comments !== 'undefined') {
                    tokens.comments = extra.comments;
                  }
                  if (typeof extra.errors !== 'undefined') {
                    tokens.errors = extra.errors;
                  }
                } catch (e) {
                  throw e;
                } finally {
                  extra = {};
                }
                return tokens;
              }
              function parse(code, options) {
                var program, toString;
                toString = String;
                if (typeof code !== 'string' && !(code instanceof String)) {
                  code = toString(code);
                }
                delegate = SyntaxTreeDelegate;
                source = code;
                index = 0;
                lineNumber = source.length > 0 ? 1 : 0;
                lineStart = 0;
                length = source.length;
                lookahead = null;
                state = {
                  allowIn: true,
                  labelSet: {},
                  inFunctionBody: false,
                  inIteration: false,
                  inSwitch: false,
                  lastCommentStart: -1
                };
                extra = {};
                if (typeof options !== 'undefined') {
                  extra.range = typeof options.range === 'boolean' && options.range;
                  extra.loc = typeof options.loc === 'boolean' && options.loc;
                  extra.attachComment = typeof options.attachComment === 'boolean' && options.attachComment;
                  if (extra.loc && options.source !== null && options.source !== undefined) {
                    extra.source = toString(options.source);
                  }
                  if (typeof options.tokens === 'boolean' && options.tokens) {
                    extra.tokens = [];
                  }
                  if (typeof options.comment === 'boolean' && options.comment) {
                    extra.comments = [];
                  }
                  if (typeof options.tolerant === 'boolean' && options.tolerant) {
                    extra.errors = [];
                  }
                  if (extra.attachComment) {
                    extra.range = true;
                    extra.comments = [];
                    extra.bottomRightStack = [];
                    extra.trailingComments = [];
                    extra.leadingComments = [];
                  }
                }
                try {
                  program = parseProgram();
                  if (typeof extra.comments !== 'undefined') {
                    program.comments = extra.comments;
                  }
                  if (typeof extra.tokens !== 'undefined') {
                    filterTokenLocation();
                    program.tokens = extra.tokens;
                  }
                  if (typeof extra.errors !== 'undefined') {
                    program.errors = extra.errors;
                  }
                } catch (e) {
                  throw e;
                } finally {
                  extra = {};
                }
                return program;
              }

              // Sync with *.json manifests.
              exports$1.version = '1.2.2';
              exports$1.tokenize = tokenize;
              exports$1.parse = parse;

              // Deep copy.
              /* istanbul ignore next */
              exports$1.Syntax = function () {
                var name,
                  types = {};
                if (typeof Object.create === 'function') {
                  types = Object.create(null);
                }
                for (name in Syntax) {
                  if (Syntax.hasOwnProperty(name)) {
                    types[name] = Syntax[name];
                  }
                }
                if (typeof Object.freeze === 'function') {
                  Object.freeze(types);
                }
                return types;
              }();
            });
            /* vim: set sw=4 ts=4 et tw=80 : */
          }, {}],
          1: [function (require, module, exports$1) {
            (function (process) {
              /* parser generated by jison 0.4.13 */
              /*
                Returns a Parser object of the following structure:
              	  Parser: {
                  yy: {}
                }
              	  Parser.prototype: {
                  yy: {},
                  trace: function(),
                  symbols_: {associative list: name ==> number},
                  terminals_: {associative list: number ==> name},
                  productions_: [...],
                  performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate, $$, _$),
                  table: [...],
                  defaultActions: {...},
                  parseError: function(str, hash),
                  parse: function(input),
              	    lexer: {
                      EOF: 1,
                      parseError: function(str, hash),
                      setInput: function(input),
                      input: function(),
                      unput: function(str),
                      more: function(),
                      less: function(n),
                      pastInput: function(),
                      upcomingInput: function(),
                      showPosition: function(),
                      test_match: function(regex_match_array, rule_index),
                      next: function(),
                      lex: function(),
                      begin: function(condition),
                      popState: function(),
                      _currentRules: function(),
                      topState: function(),
                      pushState: function(condition),
              	        options: {
                          ranges: boolean           (optional: true ==> token location info will include a .range[] member)
                          flex: boolean             (optional: true ==> flex-like lexing behaviour where the rules are tested exhaustively to find the longest match)
                          backtrack_lexer: boolean  (optional: true ==> lexer regexes are tested in order and for each matching regex the action code is invoked; the lexer terminates the scan when a token is returned by the action code)
                      },
              	        performAction: function(yy, yy_, $avoiding_name_collisions, YY_START),
                      rules: [...],
                      conditions: {associative list: name ==> set},
                  }
                }
              
                token location info (@$, _$, etc.): {
                  first_line: n,
                  last_line: n,
                  first_column: n,
                  last_column: n,
                  range: [start_number, end_number]       (where the numbers are indexes into the input string, regular zero-based)
                }
              
                the parseError function receives a 'hash' object with these members for lexer and parser errors: {
                  text:        (matched text)
                  token:       (the produced terminal token, if any)
                  line:        (yylineno)
                }
                while parser (grammar) errors will also provide these members, i.e. parser errors deliver a superset of attributes: {
                  loc:         (yylloc)
                  expected:    (string describing the set of expected tokens)
                  recoverable: (boolean: TRUE when the parser has a error recovery rule available for this particular error)
                }
              */
              var parser = function () {
                var parser = {
                  trace: function trace() {},
                  yy: {},
                  symbols_: {
                    "error": 2,
                    "JSON_PATH": 3,
                    "DOLLAR": 4,
                    "PATH_COMPONENTS": 5,
                    "LEADING_CHILD_MEMBER_EXPRESSION": 6,
                    "PATH_COMPONENT": 7,
                    "MEMBER_COMPONENT": 8,
                    "SUBSCRIPT_COMPONENT": 9,
                    "CHILD_MEMBER_COMPONENT": 10,
                    "DESCENDANT_MEMBER_COMPONENT": 11,
                    "DOT": 12,
                    "MEMBER_EXPRESSION": 13,
                    "DOT_DOT": 14,
                    "STAR": 15,
                    "IDENTIFIER": 16,
                    "SCRIPT_EXPRESSION": 17,
                    "INTEGER": 18,
                    "END": 19,
                    "CHILD_SUBSCRIPT_COMPONENT": 20,
                    "DESCENDANT_SUBSCRIPT_COMPONENT": 21,
                    "[": 22,
                    "SUBSCRIPT": 23,
                    "]": 24,
                    "SUBSCRIPT_EXPRESSION": 25,
                    "SUBSCRIPT_EXPRESSION_LIST": 26,
                    "SUBSCRIPT_EXPRESSION_LISTABLE": 27,
                    ",": 28,
                    "STRING_LITERAL": 29,
                    "ARRAY_SLICE": 30,
                    "FILTER_EXPRESSION": 31,
                    "QQ_STRING": 32,
                    "Q_STRING": 33,
                    "$accept": 0,
                    "$end": 1
                  },
                  terminals_: {
                    2: "error",
                    4: "DOLLAR",
                    12: "DOT",
                    14: "DOT_DOT",
                    15: "STAR",
                    16: "IDENTIFIER",
                    17: "SCRIPT_EXPRESSION",
                    18: "INTEGER",
                    19: "END",
                    22: "[",
                    24: "]",
                    28: ",",
                    30: "ARRAY_SLICE",
                    31: "FILTER_EXPRESSION",
                    32: "QQ_STRING",
                    33: "Q_STRING"
                  },
                  productions_: [0, [3, 1], [3, 2], [3, 1], [3, 2], [5, 1], [5, 2], [7, 1], [7, 1], [8, 1], [8, 1], [10, 2], [6, 1], [11, 2], [13, 1], [13, 1], [13, 1], [13, 1], [13, 1], [9, 1], [9, 1], [20, 3], [21, 4], [23, 1], [23, 1], [26, 1], [26, 3], [27, 1], [27, 1], [27, 1], [25, 1], [25, 1], [25, 1], [29, 1], [29, 1]],
                  performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate /* action[1] */, $$ /* vstack */, _$ /* lstack */
                  /**/) {
                    /* this == yyval */
                    if (!yy.ast) {
                      yy.ast = _ast;
                      _ast.initialize();
                    }
                    var $0 = $$.length - 1;
                    switch (yystate) {
                      case 1:
                        yy.ast.set({
                          expression: {
                            type: "root",
                            value: $$[$0]
                          }
                        });
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 2:
                        yy.ast.set({
                          expression: {
                            type: "root",
                            value: $$[$0 - 1]
                          }
                        });
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 3:
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 4:
                        yy.ast.set({
                          operation: "member",
                          scope: "child",
                          expression: {
                            type: "identifier",
                            value: $$[$0 - 1]
                          }
                        });
                        yy.ast.unshift();
                        return yy.ast.yield();
                      case 5:
                        break;
                      case 6:
                        break;
                      case 7:
                        yy.ast.set({
                          operation: "member"
                        });
                        yy.ast.push();
                        break;
                      case 8:
                        yy.ast.set({
                          operation: "subscript"
                        });
                        yy.ast.push();
                        break;
                      case 9:
                        yy.ast.set({
                          scope: "child"
                        });
                        break;
                      case 10:
                        yy.ast.set({
                          scope: "descendant"
                        });
                        break;
                      case 11:
                        break;
                      case 12:
                        yy.ast.set({
                          scope: "child",
                          operation: "member"
                        });
                        break;
                      case 13:
                        break;
                      case 14:
                        yy.ast.set({
                          expression: {
                            type: "wildcard",
                            value: $$[$0]
                          }
                        });
                        break;
                      case 15:
                        yy.ast.set({
                          expression: {
                            type: "identifier",
                            value: $$[$0]
                          }
                        });
                        break;
                      case 16:
                        yy.ast.set({
                          expression: {
                            type: "script_expression",
                            value: $$[$0]
                          }
                        });
                        break;
                      case 17:
                        yy.ast.set({
                          expression: {
                            type: "numeric_literal",
                            value: parseInt($$[$0])
                          }
                        });
                        break;
                      case 18:
                        break;
                      case 19:
                        yy.ast.set({
                          scope: "child"
                        });
                        break;
                      case 20:
                        yy.ast.set({
                          scope: "descendant"
                        });
                        break;
                      case 21:
                        break;
                      case 22:
                        break;
                      case 23:
                        break;
                      case 24:
                        $$[$0].length > 1 ? yy.ast.set({
                          expression: {
                            type: "union",
                            value: $$[$0]
                          }
                        }) : this.$ = $$[$0];
                        break;
                      case 25:
                        this.$ = [$$[$0]];
                        break;
                      case 26:
                        this.$ = $$[$0 - 2].concat($$[$0]);
                        break;
                      case 27:
                        this.$ = {
                          expression: {
                            type: "numeric_literal",
                            value: parseInt($$[$0])
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 28:
                        this.$ = {
                          expression: {
                            type: "string_literal",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 29:
                        this.$ = {
                          expression: {
                            type: "slice",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 30:
                        this.$ = {
                          expression: {
                            type: "wildcard",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 31:
                        this.$ = {
                          expression: {
                            type: "script_expression",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 32:
                        this.$ = {
                          expression: {
                            type: "filter_expression",
                            value: $$[$0]
                          }
                        };
                        yy.ast.set(this.$);
                        break;
                      case 33:
                        this.$ = $$[$0];
                        break;
                      case 34:
                        this.$ = $$[$0];
                        break;
                    }
                  },
                  table: [{
                    3: 1,
                    4: [1, 2],
                    6: 3,
                    13: 4,
                    15: [1, 5],
                    16: [1, 6],
                    17: [1, 7],
                    18: [1, 8],
                    19: [1, 9]
                  }, {
                    1: [3]
                  }, {
                    1: [2, 1],
                    5: 10,
                    7: 11,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 3],
                    5: 21,
                    7: 11,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 12],
                    12: [2, 12],
                    14: [2, 12],
                    22: [2, 12]
                  }, {
                    1: [2, 14],
                    12: [2, 14],
                    14: [2, 14],
                    22: [2, 14]
                  }, {
                    1: [2, 15],
                    12: [2, 15],
                    14: [2, 15],
                    22: [2, 15]
                  }, {
                    1: [2, 16],
                    12: [2, 16],
                    14: [2, 16],
                    22: [2, 16]
                  }, {
                    1: [2, 17],
                    12: [2, 17],
                    14: [2, 17],
                    22: [2, 17]
                  }, {
                    1: [2, 18],
                    12: [2, 18],
                    14: [2, 18],
                    22: [2, 18]
                  }, {
                    1: [2, 2],
                    7: 22,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 5],
                    12: [2, 5],
                    14: [2, 5],
                    22: [2, 5]
                  }, {
                    1: [2, 7],
                    12: [2, 7],
                    14: [2, 7],
                    22: [2, 7]
                  }, {
                    1: [2, 8],
                    12: [2, 8],
                    14: [2, 8],
                    22: [2, 8]
                  }, {
                    1: [2, 9],
                    12: [2, 9],
                    14: [2, 9],
                    22: [2, 9]
                  }, {
                    1: [2, 10],
                    12: [2, 10],
                    14: [2, 10],
                    22: [2, 10]
                  }, {
                    1: [2, 19],
                    12: [2, 19],
                    14: [2, 19],
                    22: [2, 19]
                  }, {
                    1: [2, 20],
                    12: [2, 20],
                    14: [2, 20],
                    22: [2, 20]
                  }, {
                    13: 23,
                    15: [1, 5],
                    16: [1, 6],
                    17: [1, 7],
                    18: [1, 8],
                    19: [1, 9]
                  }, {
                    13: 24,
                    15: [1, 5],
                    16: [1, 6],
                    17: [1, 7],
                    18: [1, 8],
                    19: [1, 9],
                    22: [1, 25]
                  }, {
                    15: [1, 29],
                    17: [1, 30],
                    18: [1, 33],
                    23: 26,
                    25: 27,
                    26: 28,
                    27: 32,
                    29: 34,
                    30: [1, 35],
                    31: [1, 31],
                    32: [1, 36],
                    33: [1, 37]
                  }, {
                    1: [2, 4],
                    7: 22,
                    8: 12,
                    9: 13,
                    10: 14,
                    11: 15,
                    12: [1, 18],
                    14: [1, 19],
                    20: 16,
                    21: 17,
                    22: [1, 20]
                  }, {
                    1: [2, 6],
                    12: [2, 6],
                    14: [2, 6],
                    22: [2, 6]
                  }, {
                    1: [2, 11],
                    12: [2, 11],
                    14: [2, 11],
                    22: [2, 11]
                  }, {
                    1: [2, 13],
                    12: [2, 13],
                    14: [2, 13],
                    22: [2, 13]
                  }, {
                    15: [1, 29],
                    17: [1, 30],
                    18: [1, 33],
                    23: 38,
                    25: 27,
                    26: 28,
                    27: 32,
                    29: 34,
                    30: [1, 35],
                    31: [1, 31],
                    32: [1, 36],
                    33: [1, 37]
                  }, {
                    24: [1, 39]
                  }, {
                    24: [2, 23]
                  }, {
                    24: [2, 24],
                    28: [1, 40]
                  }, {
                    24: [2, 30]
                  }, {
                    24: [2, 31]
                  }, {
                    24: [2, 32]
                  }, {
                    24: [2, 25],
                    28: [2, 25]
                  }, {
                    24: [2, 27],
                    28: [2, 27]
                  }, {
                    24: [2, 28],
                    28: [2, 28]
                  }, {
                    24: [2, 29],
                    28: [2, 29]
                  }, {
                    24: [2, 33],
                    28: [2, 33]
                  }, {
                    24: [2, 34],
                    28: [2, 34]
                  }, {
                    24: [1, 41]
                  }, {
                    1: [2, 21],
                    12: [2, 21],
                    14: [2, 21],
                    22: [2, 21]
                  }, {
                    18: [1, 33],
                    27: 42,
                    29: 34,
                    30: [1, 35],
                    32: [1, 36],
                    33: [1, 37]
                  }, {
                    1: [2, 22],
                    12: [2, 22],
                    14: [2, 22],
                    22: [2, 22]
                  }, {
                    24: [2, 26],
                    28: [2, 26]
                  }],
                  defaultActions: {
                    27: [2, 23],
                    29: [2, 30],
                    30: [2, 31],
                    31: [2, 32]
                  },
                  parseError: function parseError(str, hash) {
                    if (hash.recoverable) {
                      this.trace(str);
                    } else {
                      throw new Error(str);
                    }
                  },
                  parse: function parse(input) {
                    var self = this,
                      stack = [0],
                      vstack = [null],
                      lstack = [],
                      table = this.table,
                      yytext = '',
                      yylineno = 0,
                      yyleng = 0,
                      TERROR = 2,
                      EOF = 1;
                    var args = lstack.slice.call(arguments, 1);
                    this.lexer.setInput(input);
                    this.lexer.yy = this.yy;
                    this.yy.lexer = this.lexer;
                    this.yy.parser = this;
                    if (typeof this.lexer.yylloc == 'undefined') {
                      this.lexer.yylloc = {};
                    }
                    var yyloc = this.lexer.yylloc;
                    lstack.push(yyloc);
                    var ranges = this.lexer.options && this.lexer.options.ranges;
                    if (typeof this.yy.parseError === 'function') {
                      this.parseError = this.yy.parseError;
                    } else {
                      this.parseError = Object.getPrototypeOf(this).parseError;
                    }
                    function lex() {
                      var token;
                      token = self.lexer.lex() || EOF;
                      if (typeof token !== 'number') {
                        token = self.symbols_[token] || token;
                      }
                      return token;
                    }
                    var symbol,
                      state,
                      action,
                      r,
                      yyval = {},
                      p,
                      len,
                      newState,
                      expected;
                    while (true) {
                      state = stack[stack.length - 1];
                      if (this.defaultActions[state]) {
                        action = this.defaultActions[state];
                      } else {
                        if (symbol === null || typeof symbol == 'undefined') {
                          symbol = lex();
                        }
                        action = table[state] && table[state][symbol];
                      }
                      if (typeof action === 'undefined' || !action.length || !action[0]) {
                        var errStr = '';
                        expected = [];
                        for (p in table[state]) {
                          if (this.terminals_[p] && p > TERROR) {
                            expected.push('\'' + this.terminals_[p] + '\'');
                          }
                        }
                        if (this.lexer.showPosition) {
                          errStr = 'Parse error on line ' + (yylineno + 1) + ':\n' + this.lexer.showPosition() + '\nExpecting ' + expected.join(', ') + ', got \'' + (this.terminals_[symbol] || symbol) + '\'';
                        } else {
                          errStr = 'Parse error on line ' + (yylineno + 1) + ': Unexpected ' + (symbol == EOF ? 'end of input' : '\'' + (this.terminals_[symbol] || symbol) + '\'');
                        }
                        this.parseError(errStr, {
                          text: this.lexer.match,
                          token: this.terminals_[symbol] || symbol,
                          line: this.lexer.yylineno,
                          loc: yyloc,
                          expected: expected
                        });
                      }
                      if (action[0] instanceof Array && action.length > 1) {
                        throw new Error('Parse Error: multiple actions possible at state: ' + state + ', token: ' + symbol);
                      }
                      switch (action[0]) {
                        case 1:
                          stack.push(symbol);
                          vstack.push(this.lexer.yytext);
                          lstack.push(this.lexer.yylloc);
                          stack.push(action[1]);
                          symbol = null;
                          {
                            yyleng = this.lexer.yyleng;
                            yytext = this.lexer.yytext;
                            yylineno = this.lexer.yylineno;
                            yyloc = this.lexer.yylloc;
                          }
                          break;
                        case 2:
                          len = this.productions_[action[1]][1];
                          yyval.$ = vstack[vstack.length - len];
                          yyval._$ = {
                            first_line: lstack[lstack.length - (len || 1)].first_line,
                            last_line: lstack[lstack.length - 1].last_line,
                            first_column: lstack[lstack.length - (len || 1)].first_column,
                            last_column: lstack[lstack.length - 1].last_column
                          };
                          if (ranges) {
                            yyval._$.range = [lstack[lstack.length - (len || 1)].range[0], lstack[lstack.length - 1].range[1]];
                          }
                          r = this.performAction.apply(yyval, [yytext, yyleng, yylineno, this.yy, action[1], vstack, lstack].concat(args));
                          if (typeof r !== 'undefined') {
                            return r;
                          }
                          if (len) {
                            stack = stack.slice(0, -1 * len * 2);
                            vstack = vstack.slice(0, -1 * len);
                            lstack = lstack.slice(0, -1 * len);
                          }
                          stack.push(this.productions_[action[1]][0]);
                          vstack.push(yyval.$);
                          lstack.push(yyval._$);
                          newState = table[stack[stack.length - 2]][stack[stack.length - 1]];
                          stack.push(newState);
                          break;
                        case 3:
                          return true;
                      }
                    }
                    return true;
                  }
                };
                var _ast = {
                  initialize: function () {
                    this._nodes = [];
                    this._node = {};
                    this._stash = [];
                  },
                  set: function (props) {
                    for (var k in props) this._node[k] = props[k];
                    return this._node;
                  },
                  node: function (obj) {
                    if (arguments.length) this._node = obj;
                    return this._node;
                  },
                  push: function () {
                    this._nodes.push(this._node);
                    this._node = {};
                  },
                  unshift: function () {
                    this._nodes.unshift(this._node);
                    this._node = {};
                  },
                  yield: function () {
                    var _nodes = this._nodes;
                    this.initialize();
                    return _nodes;
                  }
                };
                /* generated by jison-lex 0.2.1 */
                var lexer = function () {
                  var lexer = {
                    EOF: 1,
                    parseError: function parseError(str, hash) {
                      if (this.yy.parser) {
                        this.yy.parser.parseError(str, hash);
                      } else {
                        throw new Error(str);
                      }
                    },
                    // resets the lexer, sets new input
                    setInput: function (input) {
                      this._input = input;
                      this._more = this._backtrack = this.done = false;
                      this.yylineno = this.yyleng = 0;
                      this.yytext = this.matched = this.match = '';
                      this.conditionStack = ['INITIAL'];
                      this.yylloc = {
                        first_line: 1,
                        first_column: 0,
                        last_line: 1,
                        last_column: 0
                      };
                      if (this.options.ranges) {
                        this.yylloc.range = [0, 0];
                      }
                      this.offset = 0;
                      return this;
                    },
                    // consumes and returns one char from the input
                    input: function () {
                      var ch = this._input[0];
                      this.yytext += ch;
                      this.yyleng++;
                      this.offset++;
                      this.match += ch;
                      this.matched += ch;
                      var lines = ch.match(/(?:\r\n?|\n).*/g);
                      if (lines) {
                        this.yylineno++;
                        this.yylloc.last_line++;
                      } else {
                        this.yylloc.last_column++;
                      }
                      if (this.options.ranges) {
                        this.yylloc.range[1]++;
                      }
                      this._input = this._input.slice(1);
                      return ch;
                    },
                    // unshifts one char (or a string) into the input
                    unput: function (ch) {
                      var len = ch.length;
                      var lines = ch.split(/(?:\r\n?|\n)/g);
                      this._input = ch + this._input;
                      this.yytext = this.yytext.substr(0, this.yytext.length - len - 1);
                      //this.yyleng -= len;
                      this.offset -= len;
                      var oldLines = this.match.split(/(?:\r\n?|\n)/g);
                      this.match = this.match.substr(0, this.match.length - 1);
                      this.matched = this.matched.substr(0, this.matched.length - 1);
                      if (lines.length - 1) {
                        this.yylineno -= lines.length - 1;
                      }
                      var r = this.yylloc.range;
                      this.yylloc = {
                        first_line: this.yylloc.first_line,
                        last_line: this.yylineno + 1,
                        first_column: this.yylloc.first_column,
                        last_column: lines ? (lines.length === oldLines.length ? this.yylloc.first_column : 0) + oldLines[oldLines.length - lines.length].length - lines[0].length : this.yylloc.first_column - len
                      };
                      if (this.options.ranges) {
                        this.yylloc.range = [r[0], r[0] + this.yyleng - len];
                      }
                      this.yyleng = this.yytext.length;
                      return this;
                    },
                    // When called from action, caches matched text and appends it on next action
                    more: function () {
                      this._more = true;
                      return this;
                    },
                    // When called from action, signals the lexer that this rule fails to match the input, so the next matching rule (regex) should be tested instead.
                    reject: function () {
                      if (this.options.backtrack_lexer) {
                        this._backtrack = true;
                      } else {
                        return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. You can only invoke reject() in the lexer when the lexer is of the backtracking persuasion (options.backtrack_lexer = true).\n' + this.showPosition(), {
                          text: "",
                          token: null,
                          line: this.yylineno
                        });
                      }
                      return this;
                    },
                    // retain first n characters of the match
                    less: function (n) {
                      this.unput(this.match.slice(n));
                    },
                    // displays already matched input, i.e. for error messages
                    pastInput: function () {
                      var past = this.matched.substr(0, this.matched.length - this.match.length);
                      return (past.length > 20 ? '...' : '') + past.substr(-20).replace(/\n/g, "");
                    },
                    // displays upcoming input, i.e. for error messages
                    upcomingInput: function () {
                      var next = this.match;
                      if (next.length < 20) {
                        next += this._input.substr(0, 20 - next.length);
                      }
                      return (next.substr(0, 20) + (next.length > 20 ? '...' : '')).replace(/\n/g, "");
                    },
                    // displays the character position where the lexing error occurred, i.e. for error messages
                    showPosition: function () {
                      var pre = this.pastInput();
                      var c = new Array(pre.length + 1).join("-");
                      return pre + this.upcomingInput() + "\n" + c + "^";
                    },
                    // test the lexed token: return FALSE when not a match, otherwise return token
                    test_match: function (match, indexed_rule) {
                      var token, lines, backup;
                      if (this.options.backtrack_lexer) {
                        // save context
                        backup = {
                          yylineno: this.yylineno,
                          yylloc: {
                            first_line: this.yylloc.first_line,
                            last_line: this.last_line,
                            first_column: this.yylloc.first_column,
                            last_column: this.yylloc.last_column
                          },
                          yytext: this.yytext,
                          match: this.match,
                          matches: this.matches,
                          matched: this.matched,
                          yyleng: this.yyleng,
                          offset: this.offset,
                          _more: this._more,
                          _input: this._input,
                          yy: this.yy,
                          conditionStack: this.conditionStack.slice(0),
                          done: this.done
                        };
                        if (this.options.ranges) {
                          backup.yylloc.range = this.yylloc.range.slice(0);
                        }
                      }
                      lines = match[0].match(/(?:\r\n?|\n).*/g);
                      if (lines) {
                        this.yylineno += lines.length;
                      }
                      this.yylloc = {
                        first_line: this.yylloc.last_line,
                        last_line: this.yylineno + 1,
                        first_column: this.yylloc.last_column,
                        last_column: lines ? lines[lines.length - 1].length - lines[lines.length - 1].match(/\r?\n?/)[0].length : this.yylloc.last_column + match[0].length
                      };
                      this.yytext += match[0];
                      this.match += match[0];
                      this.matches = match;
                      this.yyleng = this.yytext.length;
                      if (this.options.ranges) {
                        this.yylloc.range = [this.offset, this.offset += this.yyleng];
                      }
                      this._more = false;
                      this._backtrack = false;
                      this._input = this._input.slice(match[0].length);
                      this.matched += match[0];
                      token = this.performAction.call(this, this.yy, this, indexed_rule, this.conditionStack[this.conditionStack.length - 1]);
                      if (this.done && this._input) {
                        this.done = false;
                      }
                      if (token) {
                        return token;
                      } else if (this._backtrack) {
                        // recover context
                        for (var k in backup) {
                          this[k] = backup[k];
                        }
                        return false; // rule action called reject() implying the next rule should be tested instead.
                      }
                      return false;
                    },
                    // return next match in input
                    next: function () {
                      if (this.done) {
                        return this.EOF;
                      }
                      if (!this._input) {
                        this.done = true;
                      }
                      var token, match, tempMatch, index;
                      if (!this._more) {
                        this.yytext = '';
                        this.match = '';
                      }
                      var rules = this._currentRules();
                      for (var i = 0; i < rules.length; i++) {
                        tempMatch = this._input.match(this.rules[rules[i]]);
                        if (tempMatch && (!match || tempMatch[0].length > match[0].length)) {
                          match = tempMatch;
                          index = i;
                          if (this.options.backtrack_lexer) {
                            token = this.test_match(tempMatch, rules[i]);
                            if (token !== false) {
                              return token;
                            } else if (this._backtrack) {
                              match = false;
                              continue; // rule action called reject() implying a rule MISmatch.
                            } else {
                              // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
                              return false;
                            }
                          } else if (!this.options.flex) {
                            break;
                          }
                        }
                      }
                      if (match) {
                        token = this.test_match(match, rules[index]);
                        if (token !== false) {
                          return token;
                        }
                        // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
                        return false;
                      }
                      if (this._input === "") {
                        return this.EOF;
                      } else {
                        return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. Unrecognized text.\n' + this.showPosition(), {
                          text: "",
                          token: null,
                          line: this.yylineno
                        });
                      }
                    },
                    // return next match that has a token
                    lex: function lex() {
                      var r = this.next();
                      if (r) {
                        return r;
                      } else {
                        return this.lex();
                      }
                    },
                    // activates a new lexer condition state (pushes the new lexer condition state onto the condition stack)
                    begin: function begin(condition) {
                      this.conditionStack.push(condition);
                    },
                    // pop the previously active lexer condition state off the condition stack
                    popState: function popState() {
                      var n = this.conditionStack.length - 1;
                      if (n > 0) {
                        return this.conditionStack.pop();
                      } else {
                        return this.conditionStack[0];
                      }
                    },
                    // produce the lexer rule set which is active for the currently active lexer condition state
                    _currentRules: function _currentRules() {
                      if (this.conditionStack.length && this.conditionStack[this.conditionStack.length - 1]) {
                        return this.conditions[this.conditionStack[this.conditionStack.length - 1]].rules;
                      } else {
                        return this.conditions["INITIAL"].rules;
                      }
                    },
                    // return the currently active lexer condition state; when an index argument is provided it produces the N-th previous condition state, if available
                    topState: function topState(n) {
                      n = this.conditionStack.length - 1 - Math.abs(n || 0);
                      if (n >= 0) {
                        return this.conditionStack[n];
                      } else {
                        return "INITIAL";
                      }
                    },
                    // alias for begin(condition)
                    pushState: function pushState(condition) {
                      this.begin(condition);
                    },
                    // return the number of states currently on the stack
                    stateStackSize: function stateStackSize() {
                      return this.conditionStack.length;
                    },
                    options: {},
                    performAction: function anonymous(yy, yy_, $avoiding_name_collisions, YY_START
                    /**/) {
                      switch ($avoiding_name_collisions) {
                        case 0:
                          return 4;
                        case 1:
                          return 14;
                        case 2:
                          return 12;
                        case 3:
                          return 15;
                        case 4:
                          return 16;
                        case 5:
                          return 22;
                        case 6:
                          return 24;
                        case 7:
                          return 28;
                        case 8:
                          return 30;
                        case 9:
                          return 18;
                        case 10:
                          yy_.yytext = yy_.yytext.substr(1, yy_.yyleng - 2);
                          return 32;
                        case 11:
                          yy_.yytext = yy_.yytext.substr(1, yy_.yyleng - 2);
                          return 33;
                        case 12:
                          return 17;
                        case 13:
                          return 31;
                      }
                    },
                    rules: [/^(?:\$)/, /^(?:\.\.)/, /^(?:\.)/, /^(?:\*)/, /^(?:[a-zA-Z_]+[a-zA-Z0-9_]*)/, /^(?:\[)/, /^(?:\])/, /^(?:,)/, /^(?:((-?(?:0|[1-9][0-9]*)))?\:((-?(?:0|[1-9][0-9]*)))?(\:((-?(?:0|[1-9][0-9]*)))?)?)/, /^(?:(-?(?:0|[1-9][0-9]*)))/, /^(?:"(?:\\["bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^"\\])*")/, /^(?:'(?:\\['bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^'\\])*')/, /^(?:\(.+?\)(?=\]))/, /^(?:\?\(.+?\)(?=\]))/],
                    conditions: {
                      "INITIAL": {
                        "rules": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13],
                        "inclusive": true
                      }
                    }
                  };
                  return lexer;
                }();
                parser.lexer = lexer;
                function Parser() {
                  this.yy = {};
                }
                Parser.prototype = parser;
                parser.Parser = Parser;
                return new Parser();
              }();
              if (typeof require !== 'undefined' && typeof exports$1 !== 'undefined') {
                exports$1.parser = parser;
                exports$1.Parser = parser.Parser;
                exports$1.parse = function () {
                  return parser.parse.apply(parser, arguments);
                };
                exports$1.main = function commonjsMain(args) {
                  if (!args[1]) {
                    console.log('Usage: ' + args[0] + ' FILE');
                    process.exit(1);
                  }
                  var source = require('fs').readFileSync(require('path').normalize(args[1]), "utf8");
                  return exports$1.parser.parse(source);
                };
                if (typeof module !== 'undefined' && require.main === module) {
                  exports$1.main(process.argv.slice(1));
                }
              }
            }).call(this, require('_process'));
          }, {
            "_process": 14,
            "fs": 12,
            "path": 13
          }],
          2: [function (require, module, exports$1) {
            module.exports = {
              identifier: "[a-zA-Z_]+[a-zA-Z0-9_]*",
              integer: "-?(?:0|[1-9][0-9]*)",
              qq_string: "\"(?:\\\\[\"bfnrt/\\\\]|\\\\u[a-fA-F0-9]{4}|[^\"\\\\])*\"",
              q_string: "'(?:\\\\[\'bfnrt/\\\\]|\\\\u[a-fA-F0-9]{4}|[^\'\\\\])*'"
            };
          }, {}],
          3: [function (require, module, exports$1) {
            var dict = require('./dict');
            var fs = require('fs');
            var grammar = {
              lex: {
                macros: {
                  esc: "\\\\",
                  int: dict.integer
                },
                rules: [["\\$", "return 'DOLLAR'"], ["\\.\\.", "return 'DOT_DOT'"], ["\\.", "return 'DOT'"], ["\\*", "return 'STAR'"], [dict.identifier, "return 'IDENTIFIER'"], ["\\[", "return '['"], ["\\]", "return ']'"], [",", "return ','"], ["({int})?\\:({int})?(\\:({int})?)?", "return 'ARRAY_SLICE'"], ["{int}", "return 'INTEGER'"], [dict.qq_string, "yytext = yytext.substr(1,yyleng-2); return 'QQ_STRING';"], [dict.q_string, "yytext = yytext.substr(1,yyleng-2); return 'Q_STRING';"], ["\\(.+?\\)(?=\\])", "return 'SCRIPT_EXPRESSION'"], ["\\?\\(.+?\\)(?=\\])", "return 'FILTER_EXPRESSION'"]]
              },
              start: "JSON_PATH",
              bnf: {
                JSON_PATH: [['DOLLAR', 'yy.ast.set({ expression: { type: "root", value: $1 } }); yy.ast.unshift(); return yy.ast.yield()'], ['DOLLAR PATH_COMPONENTS', 'yy.ast.set({ expression: { type: "root", value: $1 } }); yy.ast.unshift(); return yy.ast.yield()'], ['LEADING_CHILD_MEMBER_EXPRESSION', 'yy.ast.unshift(); return yy.ast.yield()'], ['LEADING_CHILD_MEMBER_EXPRESSION PATH_COMPONENTS', 'yy.ast.set({ operation: "member", scope: "child", expression: { type: "identifier", value: $1 }}); yy.ast.unshift(); return yy.ast.yield()']],
                PATH_COMPONENTS: [['PATH_COMPONENT', ''], ['PATH_COMPONENTS PATH_COMPONENT', '']],
                PATH_COMPONENT: [['MEMBER_COMPONENT', 'yy.ast.set({ operation: "member" }); yy.ast.push()'], ['SUBSCRIPT_COMPONENT', 'yy.ast.set({ operation: "subscript" }); yy.ast.push() ']],
                MEMBER_COMPONENT: [['CHILD_MEMBER_COMPONENT', 'yy.ast.set({ scope: "child" })'], ['DESCENDANT_MEMBER_COMPONENT', 'yy.ast.set({ scope: "descendant" })']],
                CHILD_MEMBER_COMPONENT: [['DOT MEMBER_EXPRESSION', '']],
                LEADING_CHILD_MEMBER_EXPRESSION: [['MEMBER_EXPRESSION', 'yy.ast.set({ scope: "child", operation: "member" })']],
                DESCENDANT_MEMBER_COMPONENT: [['DOT_DOT MEMBER_EXPRESSION', '']],
                MEMBER_EXPRESSION: [['STAR', 'yy.ast.set({ expression: { type: "wildcard", value: $1 } })'], ['IDENTIFIER', 'yy.ast.set({ expression: { type: "identifier", value: $1 } })'], ['SCRIPT_EXPRESSION', 'yy.ast.set({ expression: { type: "script_expression", value: $1 } })'], ['INTEGER', 'yy.ast.set({ expression: { type: "numeric_literal", value: parseInt($1) } })'], ['END', '']],
                SUBSCRIPT_COMPONENT: [['CHILD_SUBSCRIPT_COMPONENT', 'yy.ast.set({ scope: "child" })'], ['DESCENDANT_SUBSCRIPT_COMPONENT', 'yy.ast.set({ scope: "descendant" })']],
                CHILD_SUBSCRIPT_COMPONENT: [['[ SUBSCRIPT ]', '']],
                DESCENDANT_SUBSCRIPT_COMPONENT: [['DOT_DOT [ SUBSCRIPT ]', '']],
                SUBSCRIPT: [['SUBSCRIPT_EXPRESSION', ''], ['SUBSCRIPT_EXPRESSION_LIST', '$1.length > 1? yy.ast.set({ expression: { type: "union", value: $1 } }) : $$ = $1']],
                SUBSCRIPT_EXPRESSION_LIST: [['SUBSCRIPT_EXPRESSION_LISTABLE', '$$ = [$1]'], ['SUBSCRIPT_EXPRESSION_LIST , SUBSCRIPT_EXPRESSION_LISTABLE', '$$ = $1.concat($3)']],
                SUBSCRIPT_EXPRESSION_LISTABLE: [['INTEGER', '$$ = { expression: { type: "numeric_literal", value: parseInt($1) } }; yy.ast.set($$)'], ['STRING_LITERAL', '$$ = { expression: { type: "string_literal", value: $1 } }; yy.ast.set($$)'], ['ARRAY_SLICE', '$$ = { expression: { type: "slice", value: $1 } }; yy.ast.set($$)']],
                SUBSCRIPT_EXPRESSION: [['STAR', '$$ = { expression: { type: "wildcard", value: $1 } }; yy.ast.set($$)'], ['SCRIPT_EXPRESSION', '$$ = { expression: { type: "script_expression", value: $1 } }; yy.ast.set($$)'], ['FILTER_EXPRESSION', '$$ = { expression: { type: "filter_expression", value: $1 } }; yy.ast.set($$)']],
                STRING_LITERAL: [['QQ_STRING', "$$ = $1"], ['Q_STRING', "$$ = $1"]]
              }
            };
            if (fs.readFileSync) {
              grammar.moduleInclude = fs.readFileSync(require.resolve("../include/module.js"));
              grammar.actionInclude = fs.readFileSync(require.resolve("../include/action.js"));
            }
            module.exports = grammar;
          }, {
            "./dict": 2,
            "fs": 12
          }],
          4: [function (require, module, exports$1) {
            var aesprim = require('./aesprim');
            var slice = require('./slice');
            var _evaluate = require('static-eval');
            var _uniq = require('underscore').uniq;
            var Handlers = function () {
              return this.initialize.apply(this, arguments);
            };
            Handlers.prototype.initialize = function () {
              this.traverse = traverser(true);
              this.descend = traverser();
            };
            Handlers.prototype.keys = Object.keys;
            Handlers.prototype.resolve = function (component) {
              var key = [component.operation, component.scope, component.expression.type].join('-');
              var method = this._fns[key];
              if (!method) throw new Error("couldn't resolve key: " + key);
              return method.bind(this);
            };
            Handlers.prototype.register = function (key, handler) {
              if (!handler instanceof Function) {
                throw new Error("handler must be a function");
              }
              this._fns[key] = handler;
            };
            Handlers.prototype._fns = {
              'member-child-identifier': function (component, partial) {
                var key = component.expression.value;
                var value = partial.value;
                if (value instanceof Object && key in value) {
                  return [{
                    value: value[key],
                    path: partial.path.concat(key)
                  }];
                }
              },
              'member-descendant-identifier': _traverse(function (key, value, ref) {
                return key == ref;
              }),
              'subscript-child-numeric_literal': _descend(function (key, value, ref) {
                return key === ref;
              }),
              'member-child-numeric_literal': _descend(function (key, value, ref) {
                return String(key) === String(ref);
              }),
              'subscript-descendant-numeric_literal': _traverse(function (key, value, ref) {
                return key === ref;
              }),
              'member-child-wildcard': _descend(function () {
                return true;
              }),
              'member-descendant-wildcard': _traverse(function () {
                return true;
              }),
              'subscript-descendant-wildcard': _traverse(function () {
                return true;
              }),
              'subscript-child-wildcard': _descend(function () {
                return true;
              }),
              'subscript-child-slice': function (component, partial) {
                if (is_array(partial.value)) {
                  var args = component.expression.value.split(':').map(_parse_nullable_int);
                  var values = partial.value.map(function (v, i) {
                    return {
                      value: v,
                      path: partial.path.concat(i)
                    };
                  });
                  return slice.apply(null, [values].concat(args));
                }
              },
              'subscript-child-union': function (component, partial) {
                var results = [];
                component.expression.value.forEach(function (component) {
                  var _component = {
                    operation: 'subscript',
                    scope: 'child',
                    expression: component.expression
                  };
                  var handler = this.resolve(_component);
                  var _results = handler(_component, partial);
                  if (_results) {
                    results = results.concat(_results);
                  }
                }, this);
                return unique(results);
              },
              'subscript-descendant-union': function (component, partial, count) {
                var jp = require('..');
                var self = this;
                var results = [];
                var nodes = jp.nodes(partial, '$..*').slice(1);
                nodes.forEach(function (node) {
                  if (results.length >= count) return;
                  component.expression.value.forEach(function (component) {
                    var _component = {
                      operation: 'subscript',
                      scope: 'child',
                      expression: component.expression
                    };
                    var handler = self.resolve(_component);
                    var _results = handler(_component, node);
                    results = results.concat(_results);
                  });
                });
                return unique(results);
              },
              'subscript-child-filter_expression': function (component, partial, count) {
                // slice out the expression from ?(expression)
                var src = component.expression.value.slice(2, -1);
                var ast = aesprim.parse(src).body[0].expression;
                var passable = function (key, value) {
                  return evaluate(ast, {
                    '@': value
                  });
                };
                return this.descend(partial, null, passable, count);
              },
              'subscript-descendant-filter_expression': function (component, partial, count) {
                // slice out the expression from ?(expression)
                var src = component.expression.value.slice(2, -1);
                var ast = aesprim.parse(src).body[0].expression;
                var passable = function (key, value) {
                  return evaluate(ast, {
                    '@': value
                  });
                };
                return this.traverse(partial, null, passable, count);
              },
              'subscript-child-script_expression': function (component, partial) {
                var exp = component.expression.value.slice(1, -1);
                return eval_recurse(partial, exp, '$[{{value}}]');
              },
              'member-child-script_expression': function (component, partial) {
                var exp = component.expression.value.slice(1, -1);
                return eval_recurse(partial, exp, '$.{{value}}');
              },
              'member-descendant-script_expression': function (component, partial) {
                var exp = component.expression.value.slice(1, -1);
                return eval_recurse(partial, exp, '$..value');
              }
            };
            Handlers.prototype._fns['subscript-child-string_literal'] = Handlers.prototype._fns['member-child-identifier'];
            Handlers.prototype._fns['member-descendant-numeric_literal'] = Handlers.prototype._fns['subscript-descendant-string_literal'] = Handlers.prototype._fns['member-descendant-identifier'];
            function eval_recurse(partial, src, template) {
              var jp = require('./index');
              var ast = aesprim.parse(src).body[0].expression;
              var value = evaluate(ast, {
                '@': partial.value
              });
              var path = template.replace(/\{\{\s*value\s*\}\}/g, value);
              var results = jp.nodes(partial.value, path);
              results.forEach(function (r) {
                r.path = partial.path.concat(r.path.slice(1));
              });
              return results;
            }
            function is_array(val) {
              return Array.isArray(val);
            }
            function is_object(val) {
              // is this a non-array, non-null object?
              return val && !(val instanceof Array) && val instanceof Object;
            }
            function traverser(recurse) {
              return function (partial, ref, passable, count) {
                var value = partial.value;
                var path = partial.path;
                var results = [];
                var descend = function (value, path) {
                  if (is_array(value)) {
                    value.forEach(function (element, index) {
                      if (results.length >= count) {
                        return;
                      }
                      if (passable(index, element, ref)) {
                        results.push({
                          path: path.concat(index),
                          value: element
                        });
                      }
                    });
                    value.forEach(function (element, index) {
                      if (results.length >= count) {
                        return;
                      }
                      if (recurse) {
                        descend(element, path.concat(index));
                      }
                    });
                  } else if (is_object(value)) {
                    this.keys(value).forEach(function (k) {
                      if (results.length >= count) {
                        return;
                      }
                      if (passable(k, value[k], ref)) {
                        results.push({
                          path: path.concat(k),
                          value: value[k]
                        });
                      }
                    });
                    this.keys(value).forEach(function (k) {
                      if (results.length >= count) {
                        return;
                      }
                      if (recurse) {
                        descend(value[k], path.concat(k));
                      }
                    });
                  }
                }.bind(this);
                descend(value, path);
                return results;
              };
            }
            function _descend(passable) {
              return function (component, partial, count) {
                return this.descend(partial, component.expression.value, passable, count);
              };
            }
            function _traverse(passable) {
              return function (component, partial, count) {
                return this.traverse(partial, component.expression.value, passable, count);
              };
            }
            function evaluate() {
              try {
                return _evaluate.apply(this, arguments);
              } catch (e) {}
            }
            function unique(results) {
              results = results.filter(function (d) {
                return d;
              });
              return _uniq(results, function (r) {
                return r.path.map(function (c) {
                  return String(c).replace('-', '--');
                }).join('-');
              });
            }
            function _parse_nullable_int(val) {
              var sval = String(val);
              return sval.match(/^-?[0-9]+$/) ? parseInt(sval) : null;
            }
            module.exports = Handlers;
          }, {
            "..": "jsonpath",
            "./aesprim": "./aesprim",
            "./index": 5,
            "./slice": 7,
            "static-eval": 15,
            "underscore": 12
          }],
          5: [function (require, module, exports$1) {
            var assert = require('assert');
            var dict = require('./dict');
            var Parser = require('./parser');
            var Handlers = require('./handlers');
            var JSONPath = function () {
              this.initialize.apply(this, arguments);
            };
            JSONPath.prototype.initialize = function () {
              this.parser = new Parser();
              this.handlers = new Handlers();
            };
            JSONPath.prototype.parse = function (string) {
              assert.ok(_is_string(string), "we need a path");
              return this.parser.parse(string);
            };
            JSONPath.prototype.parent = function (obj, string) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              var node = this.nodes(obj, string)[0];
              node.path.pop(); /* jshint unused:false */
              return this.value(obj, node.path);
            };
            JSONPath.prototype.apply = function (obj, string, fn) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              assert.equal(typeof fn, "function", "fn needs to be function");
              var nodes = this.nodes(obj, string).sort(function (a, b) {
                // sort nodes so we apply from the bottom up
                return b.path.length - a.path.length;
              });
              nodes.forEach(function (node) {
                var key = node.path.pop();
                var parent = this.value(obj, this.stringify(node.path));
                var val = node.value = fn.call(obj, parent[key]);
                parent[key] = val;
              }, this);
              return nodes;
            };
            JSONPath.prototype.value = function (obj, path, value) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(path, "we need a path");
              if (arguments.length >= 3) {
                var node = this.nodes(obj, path).shift();
                if (!node) return this._vivify(obj, path, value);
                var key = node.path.slice(-1).shift();
                var parent = this.parent(obj, this.stringify(node.path));
                parent[key] = value;
              }
              return this.query(obj, this.stringify(path), 1).shift();
            };
            JSONPath.prototype._vivify = function (obj, string, value) {
              var self = this;
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              var path = this.parser.parse(string).map(function (component) {
                return component.expression.value;
              });
              var setValue = function (path, value) {
                var key = path.pop();
                var node = self.value(obj, path);
                if (!node) {
                  setValue(path.concat(), typeof key === 'string' ? {} : []);
                  node = self.value(obj, path);
                }
                node[key] = value;
              };
              setValue(path, value);
              return this.query(obj, string)[0];
            };
            JSONPath.prototype.query = function (obj, string, count) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(_is_string(string), "we need a path");
              var results = this.nodes(obj, string, count).map(function (r) {
                return r.value;
              });
              return results;
            };
            JSONPath.prototype.paths = function (obj, string, count) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              var results = this.nodes(obj, string, count).map(function (r) {
                return r.path;
              });
              return results;
            };
            JSONPath.prototype.nodes = function (obj, string, count) {
              assert.ok(obj instanceof Object, "obj needs to be an object");
              assert.ok(string, "we need a path");
              if (count === 0) return [];
              var path = this.parser.parse(string);
              var handlers = this.handlers;
              var partials = [{
                path: ['$'],
                value: obj
              }];
              var matches = [];
              if (path.length && path[0].expression.type == 'root') path.shift();
              if (!path.length) return partials;
              path.forEach(function (component, index) {
                if (matches.length >= count) return;
                var handler = handlers.resolve(component);
                var _partials = [];
                partials.forEach(function (p) {
                  if (matches.length >= count) return;
                  var results = handler(component, p, count);
                  if (index == path.length - 1) {
                    // if we're through the components we're done
                    matches = matches.concat(results || []);
                  } else {
                    // otherwise accumulate and carry on through
                    _partials = _partials.concat(results || []);
                  }
                });
                partials = _partials;
              });
              return count ? matches.slice(0, count) : matches;
            };
            JSONPath.prototype.stringify = function (path) {
              assert.ok(path, "we need a path");
              var string = '$';
              var templates = {
                'descendant-member': '..{{value}}',
                'child-member': '.{{value}}',
                'descendant-subscript': '..[{{value}}]',
                'child-subscript': '[{{value}}]'
              };
              path = this._normalize(path);
              path.forEach(function (component) {
                if (component.expression.type == 'root') return;
                var key = [component.scope, component.operation].join('-');
                var template = templates[key];
                var value;
                if (component.expression.type == 'string_literal') {
                  value = JSON.stringify(component.expression.value);
                } else {
                  value = component.expression.value;
                }
                if (!template) throw new Error("couldn't find template " + key);
                string += template.replace(/{{value}}/, value);
              });
              return string;
            };
            JSONPath.prototype._normalize = function (path) {
              assert.ok(path, "we need a path");
              if (typeof path == "string") {
                return this.parser.parse(path);
              } else if (Array.isArray(path) && typeof path[0] == "string") {
                var _path = [{
                  expression: {
                    type: "root",
                    value: "$"
                  }
                }];
                path.forEach(function (component, index) {
                  if (component == '$' && index === 0) return;
                  if (typeof component == "string" && component.match("^" + dict.identifier + "$")) {
                    _path.push({
                      operation: 'member',
                      scope: 'child',
                      expression: {
                        value: component,
                        type: 'identifier'
                      }
                    });
                  } else {
                    var type = typeof component == "number" ? 'numeric_literal' : 'string_literal';
                    _path.push({
                      operation: 'subscript',
                      scope: 'child',
                      expression: {
                        value: component,
                        type: type
                      }
                    });
                  }
                });
                return _path;
              } else if (Array.isArray(path) && typeof path[0] == "object") {
                return path;
              }
              throw new Error("couldn't understand path " + path);
            };
            function _is_string(obj) {
              return Object.prototype.toString.call(obj) == '[object String]';
            }
            JSONPath.Handlers = Handlers;
            JSONPath.Parser = Parser;
            var instance = new JSONPath();
            instance.JSONPath = JSONPath;
            module.exports = instance;
          }, {
            "./dict": 2,
            "./handlers": 4,
            "./parser": 6,
            "assert": 8
          }],
          6: [function (require, module, exports$1) {
            var grammar = require('./grammar');
            var gparser = require('../generated/parser');
            var Parser = function () {
              var parser = new gparser.Parser();
              var _parseError = parser.parseError;
              parser.yy.parseError = function () {
                if (parser.yy.ast) {
                  parser.yy.ast.initialize();
                }
                _parseError.apply(parser, arguments);
              };
              return parser;
            };
            Parser.grammar = grammar;
            module.exports = Parser;
          }, {
            "../generated/parser": 1,
            "./grammar": 3
          }],
          7: [function (require, module, exports$1) {
            module.exports = function (arr, start, end, step) {
              if (typeof start == 'string') throw new Error("start cannot be a string");
              if (typeof end == 'string') throw new Error("end cannot be a string");
              if (typeof step == 'string') throw new Error("step cannot be a string");
              var len = arr.length;
              if (step === 0) throw new Error("step cannot be zero");
              step = step ? integer(step) : 1;

              // normalize negative values
              start = start < 0 ? len + start : start;
              end = end < 0 ? len + end : end;

              // default extents to extents
              start = integer(start === 0 ? 0 : !start ? step > 0 ? 0 : len - 1 : start);
              end = integer(end === 0 ? 0 : !end ? step > 0 ? len : -1 : end);

              // clamp extents
              start = step > 0 ? Math.max(0, start) : Math.min(len, start);
              end = step > 0 ? Math.min(end, len) : Math.max(-1, end);

              // return empty if extents are backwards
              if (step > 0 && end <= start) return [];
              if (step < 0 && start <= end) return [];
              var result = [];
              for (var i = start; i != end; i += step) {
                if (step < 0 && i <= end || step > 0 && i >= end) break;
                result.push(arr[i]);
              }
              return result;
            };
            function integer(val) {
              return String(val).match(/^[0-9]+$/) ? parseInt(val) : Number.isFinite(val) ? parseInt(val, 10) : 0;
            }
          }, {}],
          8: [function (require, module, exports$1) {
            // http://wiki.commonjs.org/wiki/Unit_Testing/1.0
            //
            // THIS IS NOT TESTED NOR LIKELY TO WORK OUTSIDE V8!
            //
            // Originally from narwhal.js (http://narwhaljs.org)
            // Copyright (c) 2009 Thomas Robinson <280north.com>
            //
            // Permission is hereby granted, free of charge, to any person obtaining a copy
            // of this software and associated documentation files (the 'Software'), to
            // deal in the Software without restriction, including without limitation the
            // rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
            // sell copies of the Software, and to permit persons to whom the Software is
            // furnished to do so, subject to the following conditions:
            //
            // The above copyright notice and this permission notice shall be included in
            // all copies or substantial portions of the Software.
            //
            // THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
            // IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
            // FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
            // AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
            // ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
            // WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

            // when used in node, this will actually load the util module we depend on
            // versus loading the builtin util module as happens otherwise
            // this is a bug in node module loading as far as I am concerned
            var util = require('util/');
            var pSlice = Array.prototype.slice;
            var hasOwn = Object.prototype.hasOwnProperty;

            // 1. The assert module provides functions that throw
            // AssertionError's when particular conditions are not met. The
            // assert module must conform to the following interface.

            var assert = module.exports = ok;

            // 2. The AssertionError is defined in assert.
            // new assert.AssertionError({ message: message,
            //                             actual: actual,
            //                             expected: expected })

            assert.AssertionError = function AssertionError(options) {
              this.name = 'AssertionError';
              this.actual = options.actual;
              this.expected = options.expected;
              this.operator = options.operator;
              if (options.message) {
                this.message = options.message;
                this.generatedMessage = false;
              } else {
                this.message = getMessage(this);
                this.generatedMessage = true;
              }
              var stackStartFunction = options.stackStartFunction || fail;
              if (Error.captureStackTrace) {
                Error.captureStackTrace(this, stackStartFunction);
              } else {
                // non v8 browsers so we can have a stacktrace
                var err = new Error();
                if (err.stack) {
                  var out = err.stack;

                  // try to strip useless frames
                  var fn_name = stackStartFunction.name;
                  var idx = out.indexOf('\n' + fn_name);
                  if (idx >= 0) {
                    // once we have located the function frame
                    // we need to strip out everything before it (and its line)
                    var next_line = out.indexOf('\n', idx + 1);
                    out = out.substring(next_line + 1);
                  }
                  this.stack = out;
                }
              }
            };

            // assert.AssertionError instanceof Error
            util.inherits(assert.AssertionError, Error);
            function replacer(key, value) {
              if (util.isUndefined(value)) {
                return '' + value;
              }
              if (util.isNumber(value) && !isFinite(value)) {
                return value.toString();
              }
              if (util.isFunction(value) || util.isRegExp(value)) {
                return value.toString();
              }
              return value;
            }
            function truncate(s, n) {
              if (util.isString(s)) {
                return s.length < n ? s : s.slice(0, n);
              } else {
                return s;
              }
            }
            function getMessage(self) {
              return truncate(JSON.stringify(self.actual, replacer), 128) + ' ' + self.operator + ' ' + truncate(JSON.stringify(self.expected, replacer), 128);
            }

            // At present only the three keys mentioned above are used and
            // understood by the spec. Implementations or sub modules can pass
            // other keys to the AssertionError's constructor - they will be
            // ignored.

            // 3. All of the following functions must throw an AssertionError
            // when a corresponding condition is not met, with a message that
            // may be undefined if not provided.  All assertion methods provide
            // both the actual and expected values to the assertion error for
            // display purposes.

            function fail(actual, expected, message, operator, stackStartFunction) {
              throw new assert.AssertionError({
                message: message,
                actual: actual,
                expected: expected,
                operator: operator,
                stackStartFunction: stackStartFunction
              });
            }

            // EXTENSION! allows for well behaved errors defined elsewhere.
            assert.fail = fail;

            // 4. Pure assertion tests whether a value is truthy, as determined
            // by !!guard.
            // assert.ok(guard, message_opt);
            // This statement is equivalent to assert.equal(true, !!guard,
            // message_opt);. To test strictly for the value true, use
            // assert.strictEqual(true, guard, message_opt);.

            function ok(value, message) {
              if (!value) fail(value, true, message, '==', assert.ok);
            }
            assert.ok = ok;

            // 5. The equality assertion tests shallow, coercive equality with
            // ==.
            // assert.equal(actual, expected, message_opt);

            assert.equal = function equal(actual, expected, message) {
              if (actual != expected) fail(actual, expected, message, '==', assert.equal);
            };

            // 6. The non-equality assertion tests for whether two objects are not equal
            // with != assert.notEqual(actual, expected, message_opt);

            assert.notEqual = function notEqual(actual, expected, message) {
              if (actual == expected) {
                fail(actual, expected, message, '!=', assert.notEqual);
              }
            };

            // 7. The equivalence assertion tests a deep equality relation.
            // assert.deepEqual(actual, expected, message_opt);

            assert.deepEqual = function deepEqual(actual, expected, message) {
              if (!_deepEqual(actual, expected)) {
                fail(actual, expected, message, 'deepEqual', assert.deepEqual);
              }
            };
            function _deepEqual(actual, expected) {
              // 7.1. All identical values are equivalent, as determined by ===.
              if (actual === expected) {
                return true;
              } else if (util.isBuffer(actual) && util.isBuffer(expected)) {
                if (actual.length != expected.length) return false;
                for (var i = 0; i < actual.length; i++) {
                  if (actual[i] !== expected[i]) return false;
                }
                return true;

                // 7.2. If the expected value is a Date object, the actual value is
                // equivalent if it is also a Date object that refers to the same time.
              } else if (util.isDate(actual) && util.isDate(expected)) {
                return actual.getTime() === expected.getTime();

                // 7.3 If the expected value is a RegExp object, the actual value is
                // equivalent if it is also a RegExp object with the same source and
                // properties (`global`, `multiline`, `lastIndex`, `ignoreCase`).
              } else if (util.isRegExp(actual) && util.isRegExp(expected)) {
                return actual.source === expected.source && actual.global === expected.global && actual.multiline === expected.multiline && actual.lastIndex === expected.lastIndex && actual.ignoreCase === expected.ignoreCase;

                // 7.4. Other pairs that do not both pass typeof value == 'object',
                // equivalence is determined by ==.
              } else if (!util.isObject(actual) && !util.isObject(expected)) {
                return actual == expected;

                // 7.5 For all other Object pairs, including Array objects, equivalence is
                // determined by having the same number of owned properties (as verified
                // with Object.prototype.hasOwnProperty.call), the same set of keys
                // (although not necessarily the same order), equivalent values for every
                // corresponding key, and an identical 'prototype' property. Note: this
                // accounts for both named and indexed properties on Arrays.
              } else {
                return objEquiv(actual, expected);
              }
            }
            function isArguments(object) {
              return Object.prototype.toString.call(object) == '[object Arguments]';
            }
            function objEquiv(a, b) {
              if (util.isNullOrUndefined(a) || util.isNullOrUndefined(b)) return false;
              // an identical 'prototype' property.
              if (a.prototype !== b.prototype) return false;
              // if one is a primitive, the other must be same
              if (util.isPrimitive(a) || util.isPrimitive(b)) {
                return a === b;
              }
              var aIsArgs = isArguments(a),
                bIsArgs = isArguments(b);
              if (aIsArgs && !bIsArgs || !aIsArgs && bIsArgs) return false;
              if (aIsArgs) {
                a = pSlice.call(a);
                b = pSlice.call(b);
                return _deepEqual(a, b);
              }
              var ka = objectKeys(a),
                kb = objectKeys(b),
                key,
                i;
              // having the same number of owned properties (keys incorporates
              // hasOwnProperty)
              if (ka.length != kb.length) return false;
              //the same set of keys (although not necessarily the same order),
              ka.sort();
              kb.sort();
              //~~~cheap key test
              for (i = ka.length - 1; i >= 0; i--) {
                if (ka[i] != kb[i]) return false;
              }
              //equivalent values for every corresponding key, and
              //~~~possibly expensive deep test
              for (i = ka.length - 1; i >= 0; i--) {
                key = ka[i];
                if (!_deepEqual(a[key], b[key])) return false;
              }
              return true;
            }

            // 8. The non-equivalence assertion tests for any deep inequality.
            // assert.notDeepEqual(actual, expected, message_opt);

            assert.notDeepEqual = function notDeepEqual(actual, expected, message) {
              if (_deepEqual(actual, expected)) {
                fail(actual, expected, message, 'notDeepEqual', assert.notDeepEqual);
              }
            };

            // 9. The strict equality assertion tests strict equality, as determined by ===.
            // assert.strictEqual(actual, expected, message_opt);

            assert.strictEqual = function strictEqual(actual, expected, message) {
              if (actual !== expected) {
                fail(actual, expected, message, '===', assert.strictEqual);
              }
            };

            // 10. The strict non-equality assertion tests for strict inequality, as
            // determined by !==.  assert.notStrictEqual(actual, expected, message_opt);

            assert.notStrictEqual = function notStrictEqual(actual, expected, message) {
              if (actual === expected) {
                fail(actual, expected, message, '!==', assert.notStrictEqual);
              }
            };
            function expectedException(actual, expected) {
              if (!actual || !expected) {
                return false;
              }
              if (Object.prototype.toString.call(expected) == '[object RegExp]') {
                return expected.test(actual);
              } else if (actual instanceof expected) {
                return true;
              } else if (expected.call({}, actual) === true) {
                return true;
              }
              return false;
            }
            function _throws(shouldThrow, block, expected, message) {
              var actual;
              if (util.isString(expected)) {
                message = expected;
                expected = null;
              }
              try {
                block();
              } catch (e) {
                actual = e;
              }
              message = (expected && expected.name ? ' (' + expected.name + ').' : '.') + (message ? ' ' + message : '.');
              if (shouldThrow && !actual) {
                fail(actual, expected, 'Missing expected exception' + message);
              }
              if (!shouldThrow && expectedException(actual, expected)) {
                fail(actual, expected, 'Got unwanted exception' + message);
              }
              if (shouldThrow && actual && expected && !expectedException(actual, expected) || !shouldThrow && actual) {
                throw actual;
              }
            }

            // 11. Expected to throw an error:
            // assert.throws(block, Error_opt, message_opt);

            assert.throws = function (block, /*optional*/error, /*optional*/message) {
              _throws.apply(this, [true].concat(pSlice.call(arguments)));
            };

            // EXTENSION! This is annoying to write outside this module.
            assert.doesNotThrow = function (block, /*optional*/message) {
              _throws.apply(this, [false].concat(pSlice.call(arguments)));
            };
            assert.ifError = function (err) {
              if (err) {
                throw err;
              }
            };
            var objectKeys = Object.keys || function (obj) {
              var keys = [];
              for (var key in obj) {
                if (hasOwn.call(obj, key)) keys.push(key);
              }
              return keys;
            };
          }, {
            "util/": 11
          }],
          9: [function (require, module, exports$1) {
            if (typeof Object.create === 'function') {
              // implementation from standard node.js 'util' module
              module.exports = function inherits(ctor, superCtor) {
                ctor.super_ = superCtor;
                ctor.prototype = Object.create(superCtor.prototype, {
                  constructor: {
                    value: ctor,
                    enumerable: false,
                    writable: true,
                    configurable: true
                  }
                });
              };
            } else {
              // old school shim for old browsers
              module.exports = function inherits(ctor, superCtor) {
                ctor.super_ = superCtor;
                var TempCtor = function () {};
                TempCtor.prototype = superCtor.prototype;
                ctor.prototype = new TempCtor();
                ctor.prototype.constructor = ctor;
              };
            }
          }, {}],
          10: [function (require, module, exports$1) {
            module.exports = function isBuffer(arg) {
              return arg && typeof arg === 'object' && typeof arg.copy === 'function' && typeof arg.fill === 'function' && typeof arg.readUInt8 === 'function';
            };
          }, {}],
          11: [function (require, module, exports$1) {
            (function (process, global) {
              // Copyright Joyent, Inc. and other Node contributors.
              //
              // Permission is hereby granted, free of charge, to any person obtaining a
              // copy of this software and associated documentation files (the
              // "Software"), to deal in the Software without restriction, including
              // without limitation the rights to use, copy, modify, merge, publish,
              // distribute, sublicense, and/or sell copies of the Software, and to permit
              // persons to whom the Software is furnished to do so, subject to the
              // following conditions:
              //
              // The above copyright notice and this permission notice shall be included
              // in all copies or substantial portions of the Software.
              //
              // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
              // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
              // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
              // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
              // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
              // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
              // USE OR OTHER DEALINGS IN THE SOFTWARE.

              var formatRegExp = /%[sdj%]/g;
              exports$1.format = function (f) {
                if (!isString(f)) {
                  var objects = [];
                  for (var i = 0; i < arguments.length; i++) {
                    objects.push(inspect(arguments[i]));
                  }
                  return objects.join(' ');
                }
                var i = 1;
                var args = arguments;
                var len = args.length;
                var str = String(f).replace(formatRegExp, function (x) {
                  if (x === '%%') return '%';
                  if (i >= len) return x;
                  switch (x) {
                    case '%s':
                      return String(args[i++]);
                    case '%d':
                      return Number(args[i++]);
                    case '%j':
                      try {
                        return JSON.stringify(args[i++]);
                      } catch (_) {
                        return '[Circular]';
                      }
                    default:
                      return x;
                  }
                });
                for (var x = args[i]; i < len; x = args[++i]) {
                  if (isNull(x) || !isObject(x)) {
                    str += ' ' + x;
                  } else {
                    str += ' ' + inspect(x);
                  }
                }
                return str;
              };

              // Mark that a method should not be used.
              // Returns a modified function which warns once by default.
              // If --no-deprecation is set, then it is a no-op.
              exports$1.deprecate = function (fn, msg) {
                // Allow for deprecating things in the process of starting up.
                if (isUndefined(global.process)) {
                  return function () {
                    return exports$1.deprecate(fn, msg).apply(this, arguments);
                  };
                }
                if (process.noDeprecation === true) {
                  return fn;
                }
                var warned = false;
                function deprecated() {
                  if (!warned) {
                    if (process.throwDeprecation) {
                      throw new Error(msg);
                    } else if (process.traceDeprecation) {
                      console.trace(msg);
                    } else {
                      console.error(msg);
                    }
                    warned = true;
                  }
                  return fn.apply(this, arguments);
                }
                return deprecated;
              };
              var debugs = {};
              var debugEnviron;
              exports$1.debuglog = function (set) {
                if (isUndefined(debugEnviron)) debugEnviron = process.env.NODE_DEBUG || '';
                set = set.toUpperCase();
                if (!debugs[set]) {
                  if (new RegExp('\\b' + set + '\\b', 'i').test(debugEnviron)) {
                    var pid = process.pid;
                    debugs[set] = function () {
                      var msg = exports$1.format.apply(exports$1, arguments);
                      console.error('%s %d: %s', set, pid, msg);
                    };
                  } else {
                    debugs[set] = function () {};
                  }
                }
                return debugs[set];
              };

              /**
               * Echos the value of a value. Trys to print the value out
               * in the best way possible given the different types.
               *
               * @param {Object} obj The object to print out.
               * @param {Object} opts Optional options object that alters the output.
               */
              /* legacy: obj, showHidden, depth, colors*/
              function inspect(obj, opts) {
                // default options
                var ctx = {
                  seen: [],
                  stylize: stylizeNoColor
                };
                // legacy...
                if (arguments.length >= 3) ctx.depth = arguments[2];
                if (arguments.length >= 4) ctx.colors = arguments[3];
                if (isBoolean(opts)) {
                  // legacy...
                  ctx.showHidden = opts;
                } else if (opts) {
                  // got an "options" object
                  exports$1._extend(ctx, opts);
                }
                // set default options
                if (isUndefined(ctx.showHidden)) ctx.showHidden = false;
                if (isUndefined(ctx.depth)) ctx.depth = 2;
                if (isUndefined(ctx.colors)) ctx.colors = false;
                if (isUndefined(ctx.customInspect)) ctx.customInspect = true;
                if (ctx.colors) ctx.stylize = stylizeWithColor;
                return formatValue(ctx, obj, ctx.depth);
              }
              exports$1.inspect = inspect;

              // http://en.wikipedia.org/wiki/ANSI_escape_code#graphics
              inspect.colors = {
                'bold': [1, 22],
                'italic': [3, 23],
                'underline': [4, 24],
                'inverse': [7, 27],
                'white': [37, 39],
                'grey': [90, 39],
                'black': [30, 39],
                'blue': [34, 39],
                'cyan': [36, 39],
                'green': [32, 39],
                'magenta': [35, 39],
                'red': [31, 39],
                'yellow': [33, 39]
              };

              // Don't use 'blue' not visible on cmd.exe
              inspect.styles = {
                'special': 'cyan',
                'number': 'yellow',
                'boolean': 'yellow',
                'undefined': 'grey',
                'null': 'bold',
                'string': 'green',
                'date': 'magenta',
                // "name": intentionally not styling
                'regexp': 'red'
              };
              function stylizeWithColor(str, styleType) {
                var style = inspect.styles[styleType];
                if (style) {
                  return '\u001b[' + inspect.colors[style][0] + 'm' + str + '\u001b[' + inspect.colors[style][1] + 'm';
                } else {
                  return str;
                }
              }
              function stylizeNoColor(str, styleType) {
                return str;
              }
              function arrayToHash(array) {
                var hash = {};
                array.forEach(function (val, idx) {
                  hash[val] = true;
                });
                return hash;
              }
              function formatValue(ctx, value, recurseTimes) {
                // Provide a hook for user-specified inspect functions.
                // Check that value is an object with an inspect function on it
                if (ctx.customInspect && value && isFunction(value.inspect) &&
                // Filter out the util module, it's inspect function is special
                value.inspect !== exports$1.inspect &&
                // Also filter out any prototype objects using the circular check.
                !(value.constructor && value.constructor.prototype === value)) {
                  var ret = value.inspect(recurseTimes, ctx);
                  if (!isString(ret)) {
                    ret = formatValue(ctx, ret, recurseTimes);
                  }
                  return ret;
                }

                // Primitive types cannot have properties
                var primitive = formatPrimitive(ctx, value);
                if (primitive) {
                  return primitive;
                }

                // Look up the keys of the object.
                var keys = Object.keys(value);
                var visibleKeys = arrayToHash(keys);
                if (ctx.showHidden) {
                  keys = Object.getOwnPropertyNames(value);
                }

                // IE doesn't make error fields non-enumerable
                // http://msdn.microsoft.com/en-us/library/ie/dww52sbt(v=vs.94).aspx
                if (isError(value) && (keys.indexOf('message') >= 0 || keys.indexOf('description') >= 0)) {
                  return formatError(value);
                }

                // Some type of object without properties can be shortcutted.
                if (keys.length === 0) {
                  if (isFunction(value)) {
                    var name = value.name ? ': ' + value.name : '';
                    return ctx.stylize('[Function' + name + ']', 'special');
                  }
                  if (isRegExp(value)) {
                    return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
                  }
                  if (isDate(value)) {
                    return ctx.stylize(Date.prototype.toString.call(value), 'date');
                  }
                  if (isError(value)) {
                    return formatError(value);
                  }
                }
                var base = '',
                  array = false,
                  braces = ['{', '}'];

                // Make Array say that they are Array
                if (isArray(value)) {
                  array = true;
                  braces = ['[', ']'];
                }

                // Make functions say that they are functions
                if (isFunction(value)) {
                  var n = value.name ? ': ' + value.name : '';
                  base = ' [Function' + n + ']';
                }

                // Make RegExps say that they are RegExps
                if (isRegExp(value)) {
                  base = ' ' + RegExp.prototype.toString.call(value);
                }

                // Make dates with properties first say the date
                if (isDate(value)) {
                  base = ' ' + Date.prototype.toUTCString.call(value);
                }

                // Make error with message first say the error
                if (isError(value)) {
                  base = ' ' + formatError(value);
                }
                if (keys.length === 0 && (!array || value.length == 0)) {
                  return braces[0] + base + braces[1];
                }
                if (recurseTimes < 0) {
                  if (isRegExp(value)) {
                    return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
                  } else {
                    return ctx.stylize('[Object]', 'special');
                  }
                }
                ctx.seen.push(value);
                var output;
                if (array) {
                  output = formatArray(ctx, value, recurseTimes, visibleKeys, keys);
                } else {
                  output = keys.map(function (key) {
                    return formatProperty(ctx, value, recurseTimes, visibleKeys, key, array);
                  });
                }
                ctx.seen.pop();
                return reduceToSingleString(output, base, braces);
              }
              function formatPrimitive(ctx, value) {
                if (isUndefined(value)) return ctx.stylize('undefined', 'undefined');
                if (isString(value)) {
                  var simple = '\'' + JSON.stringify(value).replace(/^"|"$/g, '').replace(/'/g, "\\'").replace(/\\"/g, '"') + '\'';
                  return ctx.stylize(simple, 'string');
                }
                if (isNumber(value)) return ctx.stylize('' + value, 'number');
                if (isBoolean(value)) return ctx.stylize('' + value, 'boolean');
                // For some reason typeof null is "object", so special case here.
                if (isNull(value)) return ctx.stylize('null', 'null');
              }
              function formatError(value) {
                return '[' + Error.prototype.toString.call(value) + ']';
              }
              function formatArray(ctx, value, recurseTimes, visibleKeys, keys) {
                var output = [];
                for (var i = 0, l = value.length; i < l; ++i) {
                  if (hasOwnProperty(value, String(i))) {
                    output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, String(i), true));
                  } else {
                    output.push('');
                  }
                }
                keys.forEach(function (key) {
                  if (!key.match(/^\d+$/)) {
                    output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, key, true));
                  }
                });
                return output;
              }
              function formatProperty(ctx, value, recurseTimes, visibleKeys, key, array) {
                var name, str, desc;
                desc = Object.getOwnPropertyDescriptor(value, key) || {
                  value: value[key]
                };
                if (desc.get) {
                  if (desc.set) {
                    str = ctx.stylize('[Getter/Setter]', 'special');
                  } else {
                    str = ctx.stylize('[Getter]', 'special');
                  }
                } else {
                  if (desc.set) {
                    str = ctx.stylize('[Setter]', 'special');
                  }
                }
                if (!hasOwnProperty(visibleKeys, key)) {
                  name = '[' + key + ']';
                }
                if (!str) {
                  if (ctx.seen.indexOf(desc.value) < 0) {
                    if (isNull(recurseTimes)) {
                      str = formatValue(ctx, desc.value, null);
                    } else {
                      str = formatValue(ctx, desc.value, recurseTimes - 1);
                    }
                    if (str.indexOf('\n') > -1) {
                      if (array) {
                        str = str.split('\n').map(function (line) {
                          return '  ' + line;
                        }).join('\n').substr(2);
                      } else {
                        str = '\n' + str.split('\n').map(function (line) {
                          return '   ' + line;
                        }).join('\n');
                      }
                    }
                  } else {
                    str = ctx.stylize('[Circular]', 'special');
                  }
                }
                if (isUndefined(name)) {
                  if (array && key.match(/^\d+$/)) {
                    return str;
                  }
                  name = JSON.stringify('' + key);
                  if (name.match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)) {
                    name = name.substr(1, name.length - 2);
                    name = ctx.stylize(name, 'name');
                  } else {
                    name = name.replace(/'/g, "\\'").replace(/\\"/g, '"').replace(/(^"|"$)/g, "'");
                    name = ctx.stylize(name, 'string');
                  }
                }
                return name + ': ' + str;
              }
              function reduceToSingleString(output, base, braces) {
                var length = output.reduce(function (prev, cur) {
                  if (cur.indexOf('\n') >= 0) ;
                  return prev + cur.replace(/\u001b\[\d\d?m/g, '').length + 1;
                }, 0);
                if (length > 60) {
                  return braces[0] + (base === '' ? '' : base + '\n ') + ' ' + output.join(',\n  ') + ' ' + braces[1];
                }
                return braces[0] + base + ' ' + output.join(', ') + ' ' + braces[1];
              }

              // NOTE: These type checking functions intentionally don't use `instanceof`
              // because it is fragile and can be easily faked with `Object.create()`.
              function isArray(ar) {
                return Array.isArray(ar);
              }
              exports$1.isArray = isArray;
              function isBoolean(arg) {
                return typeof arg === 'boolean';
              }
              exports$1.isBoolean = isBoolean;
              function isNull(arg) {
                return arg === null;
              }
              exports$1.isNull = isNull;
              function isNullOrUndefined(arg) {
                return arg == null;
              }
              exports$1.isNullOrUndefined = isNullOrUndefined;
              function isNumber(arg) {
                return typeof arg === 'number';
              }
              exports$1.isNumber = isNumber;
              function isString(arg) {
                return typeof arg === 'string';
              }
              exports$1.isString = isString;
              function isSymbol(arg) {
                return typeof arg === 'symbol';
              }
              exports$1.isSymbol = isSymbol;
              function isUndefined(arg) {
                return arg === void 0;
              }
              exports$1.isUndefined = isUndefined;
              function isRegExp(re) {
                return isObject(re) && objectToString(re) === '[object RegExp]';
              }
              exports$1.isRegExp = isRegExp;
              function isObject(arg) {
                return typeof arg === 'object' && arg !== null;
              }
              exports$1.isObject = isObject;
              function isDate(d) {
                return isObject(d) && objectToString(d) === '[object Date]';
              }
              exports$1.isDate = isDate;
              function isError(e) {
                return isObject(e) && (objectToString(e) === '[object Error]' || e instanceof Error);
              }
              exports$1.isError = isError;
              function isFunction(arg) {
                return typeof arg === 'function';
              }
              exports$1.isFunction = isFunction;
              function isPrimitive(arg) {
                return arg === null || typeof arg === 'boolean' || typeof arg === 'number' || typeof arg === 'string' || typeof arg === 'symbol' ||
                // ES6 symbol
                typeof arg === 'undefined';
              }
              exports$1.isPrimitive = isPrimitive;
              exports$1.isBuffer = require('./support/isBuffer');
              function objectToString(o) {
                return Object.prototype.toString.call(o);
              }
              function pad(n) {
                return n < 10 ? '0' + n.toString(10) : n.toString(10);
              }
              var months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

              // 26 Feb 16:19:34
              function timestamp() {
                var d = new Date();
                var time = [pad(d.getHours()), pad(d.getMinutes()), pad(d.getSeconds())].join(':');
                return [d.getDate(), months[d.getMonth()], time].join(' ');
              }

              // log is just a thin wrapper to console.log that prepends a timestamp
              exports$1.log = function () {
                console.log('%s - %s', timestamp(), exports$1.format.apply(exports$1, arguments));
              };

              /**
               * Inherit the prototype methods from one constructor into another.
               *
               * The Function.prototype.inherits from lang.js rewritten as a standalone
               * function (not on Function.prototype). NOTE: If this file is to be loaded
               * during bootstrapping this function needs to be rewritten using some native
               * functions as prototype setup using normal JavaScript does not work as
               * expected during bootstrapping (see mirror.js in r114903).
               *
               * @param {function} ctor Constructor function which needs to inherit the
               *     prototype.
               * @param {function} superCtor Constructor function to inherit prototype from.
               */
              exports$1.inherits = require('inherits');
              exports$1._extend = function (origin, add) {
                // Don't do anything if add isn't an object
                if (!add || !isObject(add)) return origin;
                var keys = Object.keys(add);
                var i = keys.length;
                while (i--) {
                  origin[keys[i]] = add[keys[i]];
                }
                return origin;
              };
              function hasOwnProperty(obj, prop) {
                return Object.prototype.hasOwnProperty.call(obj, prop);
              }
            }).call(this, require('_process'), typeof commonjsGlobal !== "undefined" ? commonjsGlobal : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {});
          }, {
            "./support/isBuffer": 10,
            "_process": 14,
            "inherits": 9
          }],
          12: [function (require, module, exports$1) {}, {}],
          13: [function (require, module, exports$1) {
            (function (process) {
              // .dirname, .basename, and .extname methods are extracted from Node.js v8.11.1,
              // backported and transplited with Babel, with backwards-compat fixes

              // Copyright Joyent, Inc. and other Node contributors.
              //
              // Permission is hereby granted, free of charge, to any person obtaining a
              // copy of this software and associated documentation files (the
              // "Software"), to deal in the Software without restriction, including
              // without limitation the rights to use, copy, modify, merge, publish,
              // distribute, sublicense, and/or sell copies of the Software, and to permit
              // persons to whom the Software is furnished to do so, subject to the
              // following conditions:
              //
              // The above copyright notice and this permission notice shall be included
              // in all copies or substantial portions of the Software.
              //
              // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
              // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
              // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
              // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
              // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
              // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
              // USE OR OTHER DEALINGS IN THE SOFTWARE.

              // resolves . and .. elements in a path array with directory names there
              // must be no slashes, empty elements, or device names (c:\) in the array
              // (so also no leading and trailing slashes - it does not distinguish
              // relative and absolute paths)
              function normalizeArray(parts, allowAboveRoot) {
                // if the path tries to go above the root, `up` ends up > 0
                var up = 0;
                for (var i = parts.length - 1; i >= 0; i--) {
                  var last = parts[i];
                  if (last === '.') {
                    parts.splice(i, 1);
                  } else if (last === '..') {
                    parts.splice(i, 1);
                    up++;
                  } else if (up) {
                    parts.splice(i, 1);
                    up--;
                  }
                }

                // if the path is allowed to go above the root, restore leading ..s
                if (allowAboveRoot) {
                  for (; up--; up) {
                    parts.unshift('..');
                  }
                }
                return parts;
              }

              // path.resolve([from ...], to)
              // posix version
              exports$1.resolve = function () {
                var resolvedPath = '',
                  resolvedAbsolute = false;
                for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
                  var path = i >= 0 ? arguments[i] : process.cwd();

                  // Skip empty and invalid entries
                  if (typeof path !== 'string') {
                    throw new TypeError('Arguments to path.resolve must be strings');
                  } else if (!path) {
                    continue;
                  }
                  resolvedPath = path + '/' + resolvedPath;
                  resolvedAbsolute = path.charAt(0) === '/';
                }

                // At this point the path should be resolved to a full absolute path, but
                // handle relative paths to be safe (might happen when process.cwd() fails)

                // Normalize the path
                resolvedPath = normalizeArray(filter(resolvedPath.split('/'), function (p) {
                  return !!p;
                }), !resolvedAbsolute).join('/');
                return (resolvedAbsolute ? '/' : '') + resolvedPath || '.';
              };

              // path.normalize(path)
              // posix version
              exports$1.normalize = function (path) {
                var isAbsolute = exports$1.isAbsolute(path),
                  trailingSlash = substr(path, -1) === '/';

                // Normalize the path
                path = normalizeArray(filter(path.split('/'), function (p) {
                  return !!p;
                }), !isAbsolute).join('/');
                if (!path && !isAbsolute) {
                  path = '.';
                }
                if (path && trailingSlash) {
                  path += '/';
                }
                return (isAbsolute ? '/' : '') + path;
              };

              // posix version
              exports$1.isAbsolute = function (path) {
                return path.charAt(0) === '/';
              };

              // posix version
              exports$1.join = function () {
                var paths = Array.prototype.slice.call(arguments, 0);
                return exports$1.normalize(filter(paths, function (p, index) {
                  if (typeof p !== 'string') {
                    throw new TypeError('Arguments to path.join must be strings');
                  }
                  return p;
                }).join('/'));
              };

              // path.relative(from, to)
              // posix version
              exports$1.relative = function (from, to) {
                from = exports$1.resolve(from).substr(1);
                to = exports$1.resolve(to).substr(1);
                function trim(arr) {
                  var start = 0;
                  for (; start < arr.length; start++) {
                    if (arr[start] !== '') break;
                  }
                  var end = arr.length - 1;
                  for (; end >= 0; end--) {
                    if (arr[end] !== '') break;
                  }
                  if (start > end) return [];
                  return arr.slice(start, end - start + 1);
                }
                var fromParts = trim(from.split('/'));
                var toParts = trim(to.split('/'));
                var length = Math.min(fromParts.length, toParts.length);
                var samePartsLength = length;
                for (var i = 0; i < length; i++) {
                  if (fromParts[i] !== toParts[i]) {
                    samePartsLength = i;
                    break;
                  }
                }
                var outputParts = [];
                for (var i = samePartsLength; i < fromParts.length; i++) {
                  outputParts.push('..');
                }
                outputParts = outputParts.concat(toParts.slice(samePartsLength));
                return outputParts.join('/');
              };
              exports$1.sep = '/';
              exports$1.delimiter = ':';
              exports$1.dirname = function (path) {
                if (typeof path !== 'string') path = path + '';
                if (path.length === 0) return '.';
                var code = path.charCodeAt(0);
                var hasRoot = code === 47 /*/*/;
                var end = -1;
                var matchedSlash = true;
                for (var i = path.length - 1; i >= 1; --i) {
                  code = path.charCodeAt(i);
                  if (code === 47 /*/*/) {
                    if (!matchedSlash) {
                      end = i;
                      break;
                    }
                  } else {
                    // We saw the first non-path separator
                    matchedSlash = false;
                  }
                }
                if (end === -1) return hasRoot ? '/' : '.';
                if (hasRoot && end === 1) {
                  // return '//';
                  // Backwards-compat fix:
                  return '/';
                }
                return path.slice(0, end);
              };
              function basename(path) {
                if (typeof path !== 'string') path = path + '';
                var start = 0;
                var end = -1;
                var matchedSlash = true;
                var i;
                for (i = path.length - 1; i >= 0; --i) {
                  if (path.charCodeAt(i) === 47 /*/*/) {
                    // If we reached a path separator that was not part of a set of path
                    // separators at the end of the string, stop now
                    if (!matchedSlash) {
                      start = i + 1;
                      break;
                    }
                  } else if (end === -1) {
                    // We saw the first non-path separator, mark this as the end of our
                    // path component
                    matchedSlash = false;
                    end = i + 1;
                  }
                }
                if (end === -1) return '';
                return path.slice(start, end);
              }

              // Uses a mixed approach for backwards-compatibility, as ext behavior changed
              // in new Node.js versions, so only basename() above is backported here
              exports$1.basename = function (path, ext) {
                var f = basename(path);
                if (ext && f.substr(-1 * ext.length) === ext) {
                  f = f.substr(0, f.length - ext.length);
                }
                return f;
              };
              exports$1.extname = function (path) {
                if (typeof path !== 'string') path = path + '';
                var startDot = -1;
                var startPart = 0;
                var end = -1;
                var matchedSlash = true;
                // Track the state of characters (if any) we see before our first dot and
                // after any path separator we find
                var preDotState = 0;
                for (var i = path.length - 1; i >= 0; --i) {
                  var code = path.charCodeAt(i);
                  if (code === 47 /*/*/) {
                    // If we reached a path separator that was not part of a set of path
                    // separators at the end of the string, stop now
                    if (!matchedSlash) {
                      startPart = i + 1;
                      break;
                    }
                    continue;
                  }
                  if (end === -1) {
                    // We saw the first non-path separator, mark this as the end of our
                    // extension
                    matchedSlash = false;
                    end = i + 1;
                  }
                  if (code === 46 /*.*/) {
                    // If this is our first dot, mark it as the start of our extension
                    if (startDot === -1) startDot = i;else if (preDotState !== 1) preDotState = 1;
                  } else if (startDot !== -1) {
                    // We saw a non-dot and non-path separator before our dot, so we should
                    // have a good chance at having a non-empty extension
                    preDotState = -1;
                  }
                }
                if (startDot === -1 || end === -1 ||
                // We saw a non-dot character immediately before the dot
                preDotState === 0 ||
                // The (right-most) trimmed path component is exactly '..'
                preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
                  return '';
                }
                return path.slice(startDot, end);
              };
              function filter(xs, f) {
                if (xs.filter) return xs.filter(f);
                var res = [];
                for (var i = 0; i < xs.length; i++) {
                  if (f(xs[i], i, xs)) res.push(xs[i]);
                }
                return res;
              }

              // String.prototype.substr - negative index don't work in IE8
              var substr = 'ab'.substr(-1) === 'b' ? function (str, start, len) {
                return str.substr(start, len);
              } : function (str, start, len) {
                if (start < 0) start = str.length + start;
                return str.substr(start, len);
              };
            }).call(this, require('_process'));
          }, {
            "_process": 14
          }],
          14: [function (require, module, exports$1) {
            // shim for using process in browser
            var process = module.exports = {};

            // cached from whatever global is present so that test runners that stub it
            // don't break things.  But we need to wrap it in a try catch in case it is
            // wrapped in strict mode code which doesn't define any globals.  It's inside a
            // function because try/catches deoptimize in certain engines.

            var cachedSetTimeout;
            var cachedClearTimeout;
            function defaultSetTimout() {
              throw new Error('setTimeout has not been defined');
            }
            function defaultClearTimeout() {
              throw new Error('clearTimeout has not been defined');
            }
            (function () {
              try {
                if (typeof setTimeout === 'function') {
                  cachedSetTimeout = setTimeout;
                } else {
                  cachedSetTimeout = defaultSetTimout;
                }
              } catch (e) {
                cachedSetTimeout = defaultSetTimout;
              }
              try {
                if (typeof clearTimeout === 'function') {
                  cachedClearTimeout = clearTimeout;
                } else {
                  cachedClearTimeout = defaultClearTimeout;
                }
              } catch (e) {
                cachedClearTimeout = defaultClearTimeout;
              }
            })();
            function runTimeout(fun) {
              if (cachedSetTimeout === setTimeout) {
                //normal enviroments in sane situations
                return setTimeout(fun, 0);
              }
              // if setTimeout wasn't available but was latter defined
              if ((cachedSetTimeout === defaultSetTimout || !cachedSetTimeout) && setTimeout) {
                cachedSetTimeout = setTimeout;
                return setTimeout(fun, 0);
              }
              try {
                // when when somebody has screwed with setTimeout but no I.E. maddness
                return cachedSetTimeout(fun, 0);
              } catch (e) {
                try {
                  // When we are in I.E. but the script has been evaled so I.E. doesn't trust the global object when called normally
                  return cachedSetTimeout.call(null, fun, 0);
                } catch (e) {
                  // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error
                  return cachedSetTimeout.call(this, fun, 0);
                }
              }
            }
            function runClearTimeout(marker) {
              if (cachedClearTimeout === clearTimeout) {
                //normal enviroments in sane situations
                return clearTimeout(marker);
              }
              // if clearTimeout wasn't available but was latter defined
              if ((cachedClearTimeout === defaultClearTimeout || !cachedClearTimeout) && clearTimeout) {
                cachedClearTimeout = clearTimeout;
                return clearTimeout(marker);
              }
              try {
                // when when somebody has screwed with setTimeout but no I.E. maddness
                return cachedClearTimeout(marker);
              } catch (e) {
                try {
                  // When we are in I.E. but the script has been evaled so I.E. doesn't  trust the global object when called normally
                  return cachedClearTimeout.call(null, marker);
                } catch (e) {
                  // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error.
                  // Some versions of I.E. have different rules for clearTimeout vs setTimeout
                  return cachedClearTimeout.call(this, marker);
                }
              }
            }
            var queue = [];
            var draining = false;
            var currentQueue;
            var queueIndex = -1;
            function cleanUpNextTick() {
              if (!draining || !currentQueue) {
                return;
              }
              draining = false;
              if (currentQueue.length) {
                queue = currentQueue.concat(queue);
              } else {
                queueIndex = -1;
              }
              if (queue.length) {
                drainQueue();
              }
            }
            function drainQueue() {
              if (draining) {
                return;
              }
              var timeout = runTimeout(cleanUpNextTick);
              draining = true;
              var len = queue.length;
              while (len) {
                currentQueue = queue;
                queue = [];
                while (++queueIndex < len) {
                  if (currentQueue) {
                    currentQueue[queueIndex].run();
                  }
                }
                queueIndex = -1;
                len = queue.length;
              }
              currentQueue = null;
              draining = false;
              runClearTimeout(timeout);
            }
            process.nextTick = function (fun) {
              var args = new Array(arguments.length - 1);
              if (arguments.length > 1) {
                for (var i = 1; i < arguments.length; i++) {
                  args[i - 1] = arguments[i];
                }
              }
              queue.push(new Item(fun, args));
              if (queue.length === 1 && !draining) {
                runTimeout(drainQueue);
              }
            };

            // v8 likes predictible objects
            function Item(fun, array) {
              this.fun = fun;
              this.array = array;
            }
            Item.prototype.run = function () {
              this.fun.apply(null, this.array);
            };
            process.title = 'browser';
            process.browser = true;
            process.env = {};
            process.argv = [];
            process.version = ''; // empty string to avoid regexp issues
            process.versions = {};
            function noop() {}
            process.on = noop;
            process.addListener = noop;
            process.once = noop;
            process.off = noop;
            process.removeListener = noop;
            process.removeAllListeners = noop;
            process.emit = noop;
            process.prependListener = noop;
            process.prependOnceListener = noop;
            process.listeners = function (name) {
              return [];
            };
            process.binding = function (name) {
              throw new Error('process.binding is not supported');
            };
            process.cwd = function () {
              return '/';
            };
            process.chdir = function (dir) {
              throw new Error('process.chdir is not supported');
            };
            process.umask = function () {
              return 0;
            };
          }, {}],
          15: [function (require, module, exports$1) {
            var unparse = require('escodegen').generate;
            module.exports = function (ast, vars) {
              if (!vars) vars = {};
              var FAIL = {};
              var result = function walk(node, scopeVars) {
                if (node.type === 'Literal') {
                  return node.value;
                } else if (node.type === 'UnaryExpression') {
                  var val = walk(node.argument);
                  if (node.operator === '+') return +val;
                  if (node.operator === '-') return -val;
                  if (node.operator === '~') return ~val;
                  if (node.operator === '!') return !val;
                  return FAIL;
                } else if (node.type === 'ArrayExpression') {
                  var xs = [];
                  for (var i = 0, l = node.elements.length; i < l; i++) {
                    var x = walk(node.elements[i]);
                    if (x === FAIL) return FAIL;
                    xs.push(x);
                  }
                  return xs;
                } else if (node.type === 'ObjectExpression') {
                  var obj = {};
                  for (var i = 0; i < node.properties.length; i++) {
                    var prop = node.properties[i];
                    var value = prop.value === null ? prop.value : walk(prop.value);
                    if (value === FAIL) return FAIL;
                    obj[prop.key.value || prop.key.name] = value;
                  }
                  return obj;
                } else if (node.type === 'BinaryExpression' || node.type === 'LogicalExpression') {
                  var l = walk(node.left);
                  if (l === FAIL) return FAIL;
                  var r = walk(node.right);
                  if (r === FAIL) return FAIL;
                  var op = node.operator;
                  if (op === '==') return l == r;
                  if (op === '===') return l === r;
                  if (op === '!=') return l != r;
                  if (op === '!==') return l !== r;
                  if (op === '+') return l + r;
                  if (op === '-') return l - r;
                  if (op === '*') return l * r;
                  if (op === '/') return l / r;
                  if (op === '%') return l % r;
                  if (op === '<') return l < r;
                  if (op === '<=') return l <= r;
                  if (op === '>') return l > r;
                  if (op === '>=') return l >= r;
                  if (op === '|') return l | r;
                  if (op === '&') return l & r;
                  if (op === '^') return l ^ r;
                  if (op === '&&') return l && r;
                  if (op === '||') return l || r;
                  return FAIL;
                } else if (node.type === 'Identifier') {
                  if ({}.hasOwnProperty.call(vars, node.name)) {
                    return vars[node.name];
                  } else return FAIL;
                } else if (node.type === 'ThisExpression') {
                  if ({}.hasOwnProperty.call(vars, 'this')) {
                    return vars['this'];
                  } else return FAIL;
                } else if (node.type === 'CallExpression') {
                  var callee = walk(node.callee);
                  if (callee === FAIL) return FAIL;
                  if (typeof callee !== 'function') return FAIL;
                  var ctx = node.callee.object ? walk(node.callee.object) : FAIL;
                  if (ctx === FAIL) ctx = null;
                  var args = [];
                  for (var i = 0, l = node.arguments.length; i < l; i++) {
                    var x = walk(node.arguments[i]);
                    if (x === FAIL) return FAIL;
                    args.push(x);
                  }
                  return callee.apply(ctx, args);
                } else if (node.type === 'MemberExpression') {
                  var obj = walk(node.object);
                  // do not allow access to methods on Function 
                  if (obj === FAIL || typeof obj == 'function') {
                    return FAIL;
                  }
                  if (node.property.type === 'Identifier') {
                    return obj[node.property.name];
                  }
                  var prop = walk(node.property);
                  if (prop === FAIL) return FAIL;
                  return obj[prop];
                } else if (node.type === 'ConditionalExpression') {
                  var val = walk(node.test);
                  if (val === FAIL) return FAIL;
                  return val ? walk(node.consequent) : walk(node.alternate);
                } else if (node.type === 'ExpressionStatement') {
                  var val = walk(node.expression);
                  if (val === FAIL) return FAIL;
                  return val;
                } else if (node.type === 'ReturnStatement') {
                  return walk(node.argument);
                } else if (node.type === 'FunctionExpression') {
                  var bodies = node.body.body;

                  // Create a "scope" for our arguments
                  var oldVars = {};
                  Object.keys(vars).forEach(function (element) {
                    oldVars[element] = vars[element];
                  });
                  for (var i = 0; i < node.params.length; i++) {
                    var key = node.params[i];
                    if (key.type == 'Identifier') {
                      vars[key.name] = null;
                    } else return FAIL;
                  }
                  for (var i in bodies) {
                    if (walk(bodies[i]) === FAIL) {
                      return FAIL;
                    }
                  }
                  // restore the vars and scope after we walk
                  vars = oldVars;
                  var keys = Object.keys(vars);
                  var vals = keys.map(function (key) {
                    return vars[key];
                  });
                  return Function(keys.join(', '), 'return ' + unparse(node)).apply(null, vals);
                } else if (node.type === 'TemplateLiteral') {
                  var str = '';
                  for (var i = 0; i < node.expressions.length; i++) {
                    str += walk(node.quasis[i]);
                    str += walk(node.expressions[i]);
                  }
                  str += walk(node.quasis[i]);
                  return str;
                } else if (node.type === 'TaggedTemplateExpression') {
                  var tag = walk(node.tag);
                  var quasi = node.quasi;
                  var strings = quasi.quasis.map(walk);
                  var values = quasi.expressions.map(walk);
                  return tag.apply(null, [strings].concat(values));
                } else if (node.type === 'TemplateElement') {
                  return node.value.cooked;
                } else return FAIL;
              }(ast);
              return result === FAIL ? undefined : result;
            };
          }, {
            "escodegen": 12
          }],
          "jsonpath": [function (require, module, exports$1) {
            module.exports = require('./lib/index');
          }, {
            "./lib/index": 5
          }]
        }, {}, ["jsonpath"])("jsonpath");
      });
    })(jsonpath);
    var jsonpathExports = jsonpath.exports;
    var jp = /*@__PURE__*/getDefaultExportFromCjs(jsonpathExports);

    /**
     * Returns the computed data section of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data section
     */
    function getComputedDataSection(data) {
      const computedData = jp.query(data, '$.Section[?(@.TOCHeading==="Chemical and Physical Properties")]' + '.Section[?(@.TOCHeading==="Computed Properties")]');
      return computedData;
    }

    /**
     * Returns a computed data section of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @param sectionName Name of the section to be returned
     * @returns Computed Data section
     */
    function getComputedPropertySection(data, sectionName) {
      const returnData = jp.query(data, `$.Section[?(@.TOCHeading==="${sectionName}")].Information[*]`)[0];
      returnData.Description = jp.query(data, `$.Section[?(@.TOCHeading==="${sectionName}")].Description`)[0];
      return returnData;
    }

    /**
     * Returns the number properties of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @param sectionName Name of the section to get the data from
     * @returns Computed Data
     */
    function getNumberProperties(data, sectionName) {
      const output = {
        value: null,
        label: null,
        reference: {
          description: null
        },
        description: null
      };
      try {
        const computationalData = getComputedDataSection(data);
        const section = getComputedPropertySection(computationalData[0], sectionName);
        output.value = section.Value.Number[0];
        output.label = sectionName;
        output.description = section.Description;
        output.reference.description = section.Reference[0];
        if (section.Value.Unit) {
          output.units = section.Value.Unit;
        }
      } catch {
        /* empty */
      }
      return output;
    }

    /**
     * Returns the complexity of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns ComputedData
     */
    function getComplexity(data) {
      const complexity = getNumberProperties(data, 'Complexity');
      return complexity;
    }

    /**
     * Returns the formal charge of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getFormalCharge(data) {
      const formalCharge = getNumberProperties(data, 'Formal Charge');
      return formalCharge;
    }

    /**
     * Returns the heavy atom count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getHeavyAtomCount(data) {
      const heavyAtomCount = getNumberProperties(data, 'Heavy Atom Count');
      return heavyAtomCount;
    }

    /**
     * Returns the hydrogen bond acceptor count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getHydrogenBondAcceptorCount(data) {
      const hydrogenBondAcceptorCount = getNumberProperties(data, 'Hydrogen Bond Acceptor Count');
      return hydrogenBondAcceptorCount;
    }

    /**
     * Returns the hydrogen bond donor count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns ComputedData
     */
    function getHydrogenBondDonorCount(data) {
      const hydrogenBondDonorCount = getNumberProperties(data, 'Hydrogen Bond Donor Count');
      return hydrogenBondDonorCount;
    }

    /**
     * Returns the rotable bond count of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns ComputedData
     */
    function getRotableBondCount(data) {
      const rotableBondCount = getNumberProperties(data, 'Rotatable Bond Count');
      return rotableBondCount;
    }

    /**
     * Returns the topological polar surface area in angstrom^2
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getTPSA(data) {
      const tpsa = getNumberProperties(data, 'Topological Polar Surface Area');
      return tpsa;
    }

    /**
     * Returns the xLogP3 of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Computed Data
     */
    function getxLogP3(data) {
      const xLogP3 = getNumberProperties(data, 'XLogP3');
      return xLogP3;
    }

    /**
     * Returns an object of computed data
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns Object of computed data
     */
    function getComputedData(data) {
      return {
        hydrogenBondDonorCount: getHydrogenBondDonorCount(data),
        hydrogenBondAcceptorCount: getHydrogenBondAcceptorCount(data),
        rotableBondCount: getRotableBondCount(data),
        heavyAtomCount: getHeavyAtomCount(data),
        formalCharge: getFormalCharge(data),
        complexity: getComplexity(data),
        tpsa: getTPSA(data),
        xLogP3: getxLogP3(data)
      };
    }

    var isMergeableObject = function isMergeableObject(value) {
      return isNonNullObject(value) && !isSpecial(value);
    };
    function isNonNullObject(value) {
      return !!value && typeof value === 'object';
    }
    function isSpecial(value) {
      var stringValue = Object.prototype.toString.call(value);
      return stringValue === '[object RegExp]' || stringValue === '[object Date]' || isReactElement(value);
    }

    // see https://github.com/facebook/react/blob/b5ac963fb791d1298e7f396236383bc955f916c1/src/isomorphic/classic/element/ReactElement.js#L21-L25
    var canUseSymbol = typeof Symbol === 'function' && Symbol.for;
    var REACT_ELEMENT_TYPE = canUseSymbol ? Symbol.for('react.element') : 0xeac7;
    function isReactElement(value) {
      return value.$$typeof === REACT_ELEMENT_TYPE;
    }
    function emptyTarget(val) {
      return Array.isArray(val) ? [] : {};
    }
    function cloneUnlessOtherwiseSpecified(value, options) {
      return options.clone !== false && options.isMergeableObject(value) ? deepmerge(emptyTarget(value), value, options) : value;
    }
    function defaultArrayMerge(target, source, options) {
      return target.concat(source).map(function (element) {
        return cloneUnlessOtherwiseSpecified(element, options);
      });
    }
    function getMergeFunction(key, options) {
      if (!options.customMerge) {
        return deepmerge;
      }
      var customMerge = options.customMerge(key);
      return typeof customMerge === 'function' ? customMerge : deepmerge;
    }
    function getEnumerableOwnPropertySymbols(target) {
      return Object.getOwnPropertySymbols ? Object.getOwnPropertySymbols(target).filter(function (symbol) {
        return Object.propertyIsEnumerable.call(target, symbol);
      }) : [];
    }
    function getKeys(target) {
      return Object.keys(target).concat(getEnumerableOwnPropertySymbols(target));
    }
    function propertyIsOnObject(object, property) {
      try {
        return property in object;
      } catch (_) {
        return false;
      }
    }

    // Protects from prototype poisoning and unexpected merging up the prototype chain.
    function propertyIsUnsafe(target, key) {
      return propertyIsOnObject(target, key) // Properties are safe to merge if they don't exist in the target yet,
      && !(Object.hasOwnProperty.call(target, key) // unsafe if they exist up the prototype chain,
      && Object.propertyIsEnumerable.call(target, key)); // and also unsafe if they're nonenumerable.
    }
    function mergeObject(target, source, options) {
      var destination = {};
      if (options.isMergeableObject(target)) {
        getKeys(target).forEach(function (key) {
          destination[key] = cloneUnlessOtherwiseSpecified(target[key], options);
        });
      }
      getKeys(source).forEach(function (key) {
        if (propertyIsUnsafe(target, key)) {
          return;
        }
        if (propertyIsOnObject(target, key) && options.isMergeableObject(source[key])) {
          destination[key] = getMergeFunction(key, options)(target[key], source[key], options);
        } else {
          destination[key] = cloneUnlessOtherwiseSpecified(source[key], options);
        }
      });
      return destination;
    }
    function deepmerge(target, source, options) {
      options = options || {};
      options.arrayMerge = options.arrayMerge || defaultArrayMerge;
      options.isMergeableObject = options.isMergeableObject || isMergeableObject;
      // cloneUnlessOtherwiseSpecified is added to `options` so that custom arrayMerge()
      // implementations can use it. The caller may not replace it.
      options.cloneUnlessOtherwiseSpecified = cloneUnlessOtherwiseSpecified;
      var sourceIsArray = Array.isArray(source);
      var targetIsArray = Array.isArray(target);
      var sourceAndTargetTypesMatch = sourceIsArray === targetIsArray;
      if (!sourceAndTargetTypesMatch) {
        return cloneUnlessOtherwiseSpecified(source, options);
      } else if (sourceIsArray) {
        return options.arrayMerge(target, source, options);
      } else {
        return mergeObject(target, source, options);
      }
    }
    deepmerge.all = function deepmergeAll(array, options) {
      if (!Array.isArray(array)) {
        throw new Error('first argument should be an array');
      }
      return array.reduce(function (prev, next) {
        return deepmerge(prev, next, options);
      }, {});
    };
    var deepmerge_1 = deepmerge;
    var cjs = deepmerge_1;
    var merge = /*@__PURE__*/getDefaultExportFromCjs(cjs);

    var quantities$1 = {exports: {}};

    /*
    The MIT License (MIT)
    Copyright © 2006-2007 Kevin C. Olbrich
    Copyright © 2010-2016 LIM SAS (http://lim.eu) - Julien Sanchez

    Permission is hereby granted, free of charge, to any person obtaining a copy of
    this software and associated documentation files (the "Software"), to deal in
    the Software without restriction, including without limitation the rights to
    use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
    of the Software, and to permit persons to whom the Software is furnished to do
    so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
    */
    var quantities = quantities$1.exports;
    (function (module, exports$1) {
      (function (global, factory) {
        module.exports = factory() ;
      })(quantities, function () {

        /**
         * Tests if a value is a string
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a string, false otherwise
         */
        function isString(value) {
          return typeof value === "string" || value instanceof String;
        }

        /*
         * Prefer stricter Number.isFinite if currently supported.
         * To be dropped when ES6 is finalized. Obsolete browsers will
         * have to use ES6 polyfills.
         */
        var isFiniteImpl = Number.isFinite || window.isFinite;
        /**
         * Tests if a value is a number
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a number, false otherwise
         */
        function isNumber(value) {
          // Number.isFinite allows not to consider NaN or '1' as numbers
          return isFiniteImpl(value);
        }

        /*
         * Identity function
         */
        function identity(value) {
          return value;
        }

        /**
         * Returns unique strings from list
         *
         * @param {string[]} strings - array of strings
         *
         *
         * @returns {string[]} a new array of strings without duplicates
         */
        function uniq(strings) {
          var seen = {};
          return strings.filter(function (item) {
            return seen.hasOwnProperty(item) ? false : seen[item] = true;
          });
        }
        function compareArray(array1, array2) {
          if (array2.length !== array1.length) {
            return false;
          }
          for (var i = 0; i < array1.length; i++) {
            if (array2[i].compareArray) {
              if (!array2[i].compareArray(array1[i])) {
                return false;
              }
            }
            if (array2[i] !== array1[i]) {
              return false;
            }
          }
          return true;
        }
        function assign(target, properties) {
          Object.keys(properties).forEach(function (key) {
            target[key] = properties[key];
          });
        }

        /**
         * Safely multiplies numbers while avoiding floating errors
         * like 0.1 * 0.1 => 0.010000000000000002
         *
         * @param {...number} numbers - numbers to multiply
         *
         * @returns {number} result
         */
        function mulSafe() {
          var result = 1,
            decimals = 0;
          for (var i = 0; i < arguments.length; i++) {
            var arg = arguments[i];
            decimals = decimals + getFractional(arg);
            result *= arg;
          }
          return decimals !== 0 ? round(result, decimals) : result;
        }

        /**
         * Safely divides two numbers while avoiding floating errors
         * like 0.3 / 0.05 => 5.999999999999999
         *
         * @returns {number} result
         * @param {number} num Numerator
         * @param {number} den Denominator
         */
        function divSafe(num, den) {
          if (den === 0) {
            throw new Error("Divide by zero");
          }
          var factor = Math.pow(10, getFractional(den));
          var invDen = factor / (factor * den);
          return mulSafe(num, invDen);
        }

        /**
         * Rounds value at the specified number of decimals
         *
         * @param {number} val - value to round
         * @param {number} decimals - number of decimals
         *
         * @returns {number} rounded number
         */
        function round(val, decimals) {
          return Math.round(val * Math.pow(10, decimals)) / Math.pow(10, decimals);
        }
        function getFractional(num) {
          // Check for NaNs or Infinities
          if (!isFinite(num)) {
            return 0;
          }

          // Faster than parsing strings
          // http://jsperf.com/count-decimals/2
          var count = 0;
          while (num % 1 !== 0) {
            num *= 10;
            count++;
          }
          return count;
        }

        /**
         * Custom error type definition
         * @constructor
         */
        function QtyError() {
          var err;
          if (!this) {
            // Allows to instantiate QtyError without new()
            err = Object.create(QtyError.prototype);
            QtyError.apply(err, arguments);
            return err;
          }
          err = Error.apply(this, arguments);
          this.name = "QtyError";
          this.message = err.message;
          this.stack = err.stack;
        }
        QtyError.prototype = Object.create(Error.prototype, {
          constructor: {
            value: QtyError
          }
        });

        /*
         * Throws incompatible units error
         * @param {string} left - units
         * @param {string} right - units incompatible with first argument
         * @throws "Incompatible units" error
         */
        function throwIncompatibleUnits(left, right) {
          throw new QtyError("Incompatible units: " + left + " and " + right);
        }
        var UNITS = {
          /* prefixes */
          "<googol>": [["googol"], 1e100, "prefix"],
          "<kibi>": [["Ki", "Kibi", "kibi"], Math.pow(2, 10), "prefix"],
          "<mebi>": [["Mi", "Mebi", "mebi"], Math.pow(2, 20), "prefix"],
          "<gibi>": [["Gi", "Gibi", "gibi"], Math.pow(2, 30), "prefix"],
          "<tebi>": [["Ti", "Tebi", "tebi"], Math.pow(2, 40), "prefix"],
          "<pebi>": [["Pi", "Pebi", "pebi"], Math.pow(2, 50), "prefix"],
          "<exi>": [["Ei", "Exi", "exi"], Math.pow(2, 60), "prefix"],
          "<zebi>": [["Zi", "Zebi", "zebi"], Math.pow(2, 70), "prefix"],
          "<yebi>": [["Yi", "Yebi", "yebi"], Math.pow(2, 80), "prefix"],
          "<yotta>": [["Y", "Yotta", "yotta"], 1e24, "prefix"],
          "<zetta>": [["Z", "Zetta", "zetta"], 1e21, "prefix"],
          "<exa>": [["E", "Exa", "exa"], 1e18, "prefix"],
          "<peta>": [["P", "Peta", "peta"], 1e15, "prefix"],
          "<tera>": [["T", "Tera", "tera"], 1e12, "prefix"],
          "<giga>": [["G", "Giga", "giga"], 1e9, "prefix"],
          "<mega>": [["M", "Mega", "mega"], 1e6, "prefix"],
          "<kilo>": [["k", "kilo"], 1e3, "prefix"],
          "<hecto>": [["h", "Hecto", "hecto"], 1e2, "prefix"],
          "<deca>": [["da", "Deca", "deca", "deka"], 1e1, "prefix"],
          "<deci>": [["d", "Deci", "deci"], 1e-1, "prefix"],
          "<centi>": [["c", "Centi", "centi"], 1e-2, "prefix"],
          "<milli>": [["m", "Milli", "milli"], 1e-3, "prefix"],
          "<micro>": [["u", "\u03BC" /*µ as greek letter*/, "\u00B5" /*µ as micro sign*/, "Micro", "mc", "micro"], 1e-6, "prefix"],
          "<nano>": [["n", "Nano", "nano"], 1e-9, "prefix"],
          "<pico>": [["p", "Pico", "pico"], 1e-12, "prefix"],
          "<femto>": [["f", "Femto", "femto"], 1e-15, "prefix"],
          "<atto>": [["a", "Atto", "atto"], 1e-18, "prefix"],
          "<zepto>": [["z", "Zepto", "zepto"], 1e-21, "prefix"],
          "<yocto>": [["y", "Yocto", "yocto"], 1e-24, "prefix"],
          "<1>": [["1", "<1>"], 1, ""],
          /* length units */
          "<meter>": [["m", "meter", "meters", "metre", "metres"], 1.0, "length", ["<meter>"]],
          "<inch>": [["in", "inch", "inches", "\""], 0.0254, "length", ["<meter>"]],
          "<foot>": [["ft", "foot", "feet", "'"], 0.3048, "length", ["<meter>"]],
          "<yard>": [["yd", "yard", "yards"], 0.9144, "length", ["<meter>"]],
          "<mile>": [["mi", "mile", "miles"], 1609.344, "length", ["<meter>"]],
          "<naut-mile>": [["nmi", "naut-mile"], 1852, "length", ["<meter>"]],
          "<league>": [["league", "leagues"], 4828, "length", ["<meter>"]],
          "<furlong>": [["furlong", "furlongs"], 201.2, "length", ["<meter>"]],
          "<rod>": [["rd", "rod", "rods"], 5.029, "length", ["<meter>"]],
          "<mil>": [["mil", "mils"], 0.0000254, "length", ["<meter>"]],
          "<angstrom>": [["ang", "angstrom", "angstroms"], 1e-10, "length", ["<meter>"]],
          "<fathom>": [["fathom", "fathoms"], 1.829, "length", ["<meter>"]],
          "<pica>": [["pica", "picas"], 0.00423333333, "length", ["<meter>"]],
          "<point>": [["pt", "point", "points"], 0.000352777778, "length", ["<meter>"]],
          "<redshift>": [["z", "red-shift", "redshift"], 1.302773e26, "length", ["<meter>"]],
          "<AU>": [["AU", "astronomical-unit"], 149597900000, "length", ["<meter>"]],
          "<light-second>": [["ls", "light-second"], 299792500, "length", ["<meter>"]],
          "<light-minute>": [["lmin", "light-minute"], 17987550000, "length", ["<meter>"]],
          "<light-year>": [["ly", "light-year"], 9460528000000000, "length", ["<meter>"]],
          "<parsec>": [["pc", "parsec", "parsecs"], 30856780000000000, "length", ["<meter>"]],
          "<datamile>": [["DM", "datamile"], 1828.8, "length", ["<meter>"]],
          /* mass */
          "<kilogram>": [["kg", "kilogram", "kilograms"], 1.0, "mass", ["<kilogram>"]],
          "<AMU>": [["u", "AMU", "amu"], 1.660538921e-27, "mass", ["<kilogram>"]],
          "<dalton>": [["Da", "Dalton", "Daltons", "dalton", "daltons"], 1.660538921e-27, "mass", ["<kilogram>"]],
          "<slug>": [["slug", "slugs"], 14.5939029, "mass", ["<kilogram>"]],
          "<short-ton>": [["tn", "ton", "short-ton"], 907.18474, "mass", ["<kilogram>"]],
          "<metric-ton>": [["t", "tonne", "metric-ton"], 1000, "mass", ["<kilogram>"]],
          "<carat>": [["ct", "carat", "carats"], 0.0002, "mass", ["<kilogram>"]],
          "<pound>": [["lbs", "lb", "pound", "pounds", "#"], 0.45359237, "mass", ["<kilogram>"]],
          "<ounce>": [["oz", "ounce", "ounces"], 0.0283495231, "mass", ["<kilogram>"]],
          "<gram>": [["g", "gram", "grams", "gramme", "grammes"], 1e-3, "mass", ["<kilogram>"]],
          "<grain>": [["grain", "grains", "gr"], 6.479891e-5, "mass", ["<kilogram>"]],
          "<dram>": [["dram", "drams", "dr"], 0.0017718452, "mass", ["<kilogram>"]],
          "<stone>": [["stone", "stones", "st"], 6.35029318, "mass", ["<kilogram>"]],
          /* area */
          "<hectare>": [["hectare"], 10000, "area", ["<meter>", "<meter>"]],
          "<acre>": [["acre", "acres"], 4046.85642, "area", ["<meter>", "<meter>"]],
          "<sqft>": [["sqft"], 1, "area", ["<foot>", "<foot>"]],
          /* volume */
          "<liter>": [["l", "L", "liter", "liters", "litre", "litres"], 0.001, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<gallon>": [["gal", "gallon", "gallons"], 0.0037854118, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<gallon-imp>": [["galimp", "gallon-imp", "gallons-imp"], 0.0045460900, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<quart>": [["qt", "quart", "quarts"], 0.00094635295, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<pint>": [["pt", "pint", "pints"], 0.000473176475, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<pint-imp>": [["ptimp", "pint-imp", "pints-imp"], 5.6826125e-4, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<cup>": [["cu", "cup", "cups"], 0.000236588238, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<fluid-ounce>": [["floz", "fluid-ounce", "fluid-ounces"], 2.95735297e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<fluid-ounce-imp>": [["flozimp", "floz-imp", "fluid-ounce-imp", "fluid-ounces-imp"], 2.84130625e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<tablespoon>": [["tb", "tbsp", "tbs", "tablespoon", "tablespoons"], 1.47867648e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<teaspoon>": [["tsp", "teaspoon", "teaspoons"], 4.92892161e-6, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<bushel>": [["bu", "bsh", "bushel", "bushels"], 0.035239072, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<oilbarrel>": [["bbl", "oilbarrel", "oilbarrels", "oil-barrel", "oil-barrels"], 0.158987294928, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<beerbarrel>": [["bl", "bl-us", "beerbarrel", "beerbarrels", "beer-barrel", "beer-barrels"], 0.1173477658, "volume", ["<meter>", "<meter>", "<meter>"]],
          "<beerbarrel-imp>": [["blimp", "bl-imp", "beerbarrel-imp", "beerbarrels-imp", "beer-barrel-imp", "beer-barrels-imp"], 0.16365924, "volume", ["<meter>", "<meter>", "<meter>"]],
          /* speed */
          "<kph>": [["kph"], 0.277777778, "speed", ["<meter>"], ["<second>"]],
          "<mph>": [["mph"], 0.44704, "speed", ["<meter>"], ["<second>"]],
          "<knot>": [["kt", "kn", "kts", "knot", "knots"], 0.514444444, "speed", ["<meter>"], ["<second>"]],
          "<fps>": [["fps"], 0.3048, "speed", ["<meter>"], ["<second>"]],
          /* acceleration */
          "<gee>": [["gee"], 9.80665, "acceleration", ["<meter>"], ["<second>", "<second>"]],
          "<Gal>": [["Gal"], 1e-2, "acceleration", ["<meter>"], ["<second>", "<second>"]],
          /* temperature_difference */
          "<kelvin>": [["degK", "kelvin"], 1.0, "temperature", ["<kelvin>"]],
          "<celsius>": [["degC", "celsius", "celsius", "centigrade"], 1.0, "temperature", ["<kelvin>"]],
          "<fahrenheit>": [["degF", "fahrenheit"], 5 / 9, "temperature", ["<kelvin>"]],
          "<rankine>": [["degR", "rankine"], 5 / 9, "temperature", ["<kelvin>"]],
          "<temp-K>": [["tempK", "temp-K"], 1.0, "temperature", ["<temp-K>"]],
          "<temp-C>": [["tempC", "temp-C"], 1.0, "temperature", ["<temp-K>"]],
          "<temp-F>": [["tempF", "temp-F"], 5 / 9, "temperature", ["<temp-K>"]],
          "<temp-R>": [["tempR", "temp-R"], 5 / 9, "temperature", ["<temp-K>"]],
          /* time */
          "<second>": [["s", "sec", "secs", "second", "seconds"], 1.0, "time", ["<second>"]],
          "<minute>": [["min", "mins", "minute", "minutes"], 60.0, "time", ["<second>"]],
          "<hour>": [["h", "hr", "hrs", "hour", "hours"], 3600.0, "time", ["<second>"]],
          "<day>": [["d", "day", "days"], 3600 * 24, "time", ["<second>"]],
          "<week>": [["wk", "week", "weeks"], 7 * 3600 * 24, "time", ["<second>"]],
          "<fortnight>": [["fortnight", "fortnights"], 1209600, "time", ["<second>"]],
          "<year>": [["y", "yr", "year", "years", "annum"], 31556926, "time", ["<second>"]],
          "<decade>": [["decade", "decades"], 315569260, "time", ["<second>"]],
          "<century>": [["century", "centuries"], 3155692600, "time", ["<second>"]],
          /* pressure */
          "<pascal>": [["Pa", "pascal", "Pascal"], 1.0, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<bar>": [["bar", "bars"], 100000, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<mmHg>": [["mmHg"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<inHg>": [["inHg"], 3386.3881472, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<torr>": [["torr"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<atm>": [["atm", "ATM", "atmosphere", "atmospheres"], 101325, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<psi>": [["psi"], 6894.76, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<cmh2o>": [["cmH2O", "cmh2o"], 98.0638, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          "<inh2o>": [["inH2O", "inh2o"], 249.082052, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
          /* viscosity */
          "<poise>": [["P", "poise"], 0.1, "viscosity", ["<kilogram>"], ["<meter>", "<second>"]],
          "<stokes>": [["St", "stokes"], 1e-4, "viscosity", ["<meter>", "<meter>"], ["<second>"]],
          /* substance */
          "<mole>": [["mol", "mole"], 1.0, "substance", ["<mole>"]],
          /* molar_concentration */
          "<molar>": [["M", "molar"], 1000, "molar_concentration", ["<mole>"], ["<meter>", "<meter>", "<meter>"]],
          "<wtpercent>": [["wt%", "wtpercent"], 10, "molar_concentration", ["<kilogram>"], ["<meter>", "<meter>", "<meter>"]],
          /* activity */
          "<katal>": [["kat", "katal", "Katal"], 1.0, "activity", ["<mole>"], ["<second>"]],
          "<unit>": [["U", "enzUnit", "unit"], 16.667e-16, "activity", ["<mole>"], ["<second>"]],
          /* capacitance */
          "<farad>": [["F", "farad", "Farad"], 1.0, "capacitance", ["<second>", "<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<meter>", "<meter>", "<kilogram>"]],
          /* charge */
          "<coulomb>": [["C", "coulomb", "Coulomb"], 1.0, "charge", ["<ampere>", "<second>"]],
          "<Ah>": [["Ah"], 3600, "charge", ["<ampere>", "<second>"]],
          /* current */
          "<ampere>": [["A", "Ampere", "ampere", "amp", "amps"], 1.0, "current", ["<ampere>"]],
          /* conductance */
          "<siemens>": [["S", "Siemens", "siemens"], 1.0, "conductance", ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<kilogram>", "<meter>", "<meter>"]],
          /* inductance */
          "<henry>": [["H", "Henry", "henry"], 1.0, "inductance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>", "<ampere>"]],
          /* potential */
          "<volt>": [["V", "Volt", "volt", "volts"], 1.0, "potential", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>"]],
          /* resistance */
          "<ohm>": [["Ohm", "ohm", "\u03A9" /*Ω as greek letter*/, "\u2126" /*Ω as ohm sign*/], 1.0, "resistance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"]],
          /* magnetism */
          "<weber>": [["Wb", "weber", "webers"], 1.0, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<tesla>": [["T", "tesla", "teslas"], 1.0, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<gauss>": [["G", "gauss"], 1e-4, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<maxwell>": [["Mx", "maxwell", "maxwells"], 1e-8, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
          "<oersted>": [["Oe", "oersted", "oersteds"], 250.0 / Math.PI, "magnetism", ["<ampere>"], ["<meter>"]],
          /* energy */
          "<joule>": [["J", "joule", "Joule", "joules", "Joules"], 1.0, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<erg>": [["erg", "ergs"], 1e-7, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<btu>": [["BTU", "btu", "BTUs"], 1055.056, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<calorie>": [["cal", "calorie", "calories"], 4.18400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<Calorie>": [["Cal", "Calorie", "Calories"], 4184.00, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<therm-US>": [["th", "therm", "therms", "Therm", "therm-US"], 105480400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<Wh>": [["Wh"], 3600, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          "<electronvolt>": [["eV", "electronvolt", "electronvolts"], 1.602176634E-19, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
          /* force */
          "<newton>": [["N", "Newton", "newton"], 1.0, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
          "<dyne>": [["dyn", "dyne"], 1e-5, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
          "<pound-force>": [["lbf", "pound-force"], 4.448222, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
          /* frequency */
          "<hertz>": [["Hz", "hertz", "Hertz"], 1.0, "frequency", ["<1>"], ["<second>"]],
          /* angle */
          "<radian>": [["rad", "radian", "radians"], 1.0, "angle", ["<radian>"]],
          "<degree>": [["deg", "degree", "degrees"], Math.PI / 180.0, "angle", ["<radian>"]],
          "<arcminute>": [["arcmin", "arcminute", "arcminutes"], Math.PI / 10800.0, "angle", ["<radian>"]],
          "<arcsecond>": [["arcsec", "arcsecond", "arcseconds"], Math.PI / 648000.0, "angle", ["<radian>"]],
          "<gradian>": [["gon", "grad", "gradian", "grads"], Math.PI / 200.0, "angle", ["<radian>"]],
          "<steradian>": [["sr", "steradian", "steradians"], 1.0, "solid_angle", ["<steradian>"]],
          /* rotation */
          "<rotation>": [["rotation"], 2.0 * Math.PI, "angle", ["<radian>"]],
          "<rpm>": [["rpm"], 2.0 * Math.PI / 60.0, "angular_velocity", ["<radian>"], ["<second>"]],
          /* information */
          "<byte>": [["B", "byte", "bytes"], 1.0, "information", ["<byte>"]],
          "<bit>": [["b", "bit", "bits"], 0.125, "information", ["<byte>"]],
          /* information rate */
          "<Bps>": [["Bps"], 1.0, "information_rate", ["<byte>"], ["<second>"]],
          "<bps>": [["bps"], 0.125, "information_rate", ["<byte>"], ["<second>"]],
          /* currency */
          "<dollar>": [["USD", "dollar"], 1.0, "currency", ["<dollar>"]],
          "<cents>": [["cents"], 0.01, "currency", ["<dollar>"]],
          /* luminosity */
          "<candela>": [["cd", "candela"], 1.0, "luminosity", ["<candela>"]],
          "<lumen>": [["lm", "lumen"], 1.0, "luminous_power", ["<candela>", "<steradian>"]],
          "<lux>": [["lux"], 1.0, "illuminance", ["<candela>", "<steradian>"], ["<meter>", "<meter>"]],
          /* power */
          "<watt>": [["W", "watt", "watts"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          "<volt-ampere>": [["VA", "volt-ampere"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          "<volt-ampere-reactive>": [["var", "Var", "VAr", "VAR", "volt-ampere-reactive"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          "<horsepower>": [["hp", "horsepower"], 745.699872, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
          /* radiation */
          "<gray>": [["Gy", "gray", "grays"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
          "<roentgen>": [["R", "roentgen"], 0.009330, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
          "<sievert>": [["Sv", "sievert", "sieverts"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
          "<becquerel>": [["Bq", "becquerel", "becquerels"], 1.0, "radiation", ["<1>"], ["<second>"]],
          "<curie>": [["Ci", "curie", "curies"], 3.7e10, "radiation", ["<1>"], ["<second>"]],
          /* rate */
          "<cpm>": [["cpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
          "<dpm>": [["dpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
          "<bpm>": [["bpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
          /* resolution / typography */
          "<dot>": [["dot", "dots"], 1, "resolution", ["<each>"]],
          "<pixel>": [["pixel", "px"], 1, "resolution", ["<each>"]],
          "<ppi>": [["ppi"], 1, "resolution", ["<pixel>"], ["<inch>"]],
          "<dpi>": [["dpi"], 1, "typography", ["<dot>"], ["<inch>"]],
          /* other */
          "<cell>": [["cells", "cell"], 1, "counting", ["<each>"]],
          "<each>": [["each"], 1.0, "counting", ["<each>"]],
          "<count>": [["count"], 1.0, "counting", ["<each>"]],
          "<base-pair>": [["bp", "base-pair"], 1.0, "counting", ["<each>"]],
          "<nucleotide>": [["nt", "nucleotide"], 1.0, "counting", ["<each>"]],
          "<molecule>": [["molecule", "molecules"], 1.0, "counting", ["<1>"]],
          "<dozen>": [["doz", "dz", "dozen"], 12.0, "prefix_only", ["<each>"]],
          "<percent>": [["%", "percent"], 0.01, "prefix_only", ["<1>"]],
          "<ppm>": [["ppm"], 1e-6, "prefix_only", ["<1>"]],
          "<ppb>": [["ppb"], 1e-9, "prefix_only", ["<1>"]],
          "<ppt>": [["ppt"], 1e-12, "prefix_only", ["<1>"]],
          "<ppq>": [["ppq"], 1e-15, "prefix_only", ["<1>"]],
          "<gross>": [["gr", "gross"], 144.0, "prefix_only", ["<dozen>", "<dozen>"]],
          "<decibel>": [["dB", "decibel", "decibels"], 1.0, "logarithmic", ["<decibel>"]]
        };
        var BASE_UNITS = ["<meter>", "<kilogram>", "<second>", "<mole>", "<ampere>", "<radian>", "<kelvin>", "<temp-K>", "<byte>", "<dollar>", "<candela>", "<each>", "<steradian>", "<decibel>"];
        var UNITY = "<1>";
        var UNITY_ARRAY = [UNITY];

        // Setup

        /**
         * Asserts unit definition is valid
         *
         * @param {string} unitDef - Name of unit to test
         * @param {Object} definition - Definition of unit to test
         *
         * @returns {void}
         * @throws {QtyError} if unit definition is not valid
         */
        function validateUnitDefinition(unitDef, definition) {
          var scalar = definition[1];
          var numerator = definition[3] || [];
          var denominator = definition[4] || [];
          if (!isNumber(scalar)) {
            throw new QtyError(unitDef + ": Invalid unit definition. " + "'scalar' must be a number");
          }
          numerator.forEach(function (unit) {
            if (UNITS[unit] === undefined) {
              throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'numerator' is not recognized");
            }
          });
          denominator.forEach(function (unit) {
            if (UNITS[unit] === undefined) {
              throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'denominator' is not recognized");
            }
          });
        }
        var PREFIX_VALUES = {};
        var PREFIX_MAP = {};
        var UNIT_VALUES = {};
        var UNIT_MAP = {};
        var OUTPUT_MAP = {};
        for (var unitDef in UNITS) {
          if (UNITS.hasOwnProperty(unitDef)) {
            var definition = UNITS[unitDef];
            if (definition[2] === "prefix") {
              PREFIX_VALUES[unitDef] = definition[1];
              for (var i = 0; i < definition[0].length; i++) {
                PREFIX_MAP[definition[0][i]] = unitDef;
              }
            } else {
              validateUnitDefinition(unitDef, definition);
              UNIT_VALUES[unitDef] = {
                scalar: definition[1],
                numerator: definition[3],
                denominator: definition[4]
              };
              for (var j = 0; j < definition[0].length; j++) {
                UNIT_MAP[definition[0][j]] = unitDef;
              }
            }
            OUTPUT_MAP[unitDef] = definition[0][0];
          }
        }

        /**
         * Returns a list of available units of kind
         *
         * @param {string} [kind] - kind of units
         * @returns {array} names of units
         * @throws {QtyError} if kind is unknown
         */
        function getUnits(kind) {
          var i;
          var units = [];
          var unitKeys = Object.keys(UNITS);
          if (typeof kind === "undefined") {
            for (i = 0; i < unitKeys.length; i++) {
              if (["", "prefix"].indexOf(UNITS[unitKeys[i]][2]) === -1) {
                units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
              }
            }
          } else if (this.getKinds().indexOf(kind) === -1) {
            throw new QtyError("Kind not recognized");
          } else {
            for (i = 0; i < unitKeys.length; i++) {
              if (UNITS[unitKeys[i]][2] === kind) {
                units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
              }
            }
          }
          return units.sort(function (a, b) {
            if (a.toLowerCase() < b.toLowerCase()) {
              return -1;
            }
            if (a.toLowerCase() > b.toLowerCase()) {
              return 1;
            }
            return 0;
          });
        }

        /**
         * Returns a list of alternative names for a unit
         *
         * @param {string} unitName - name of unit
         * @returns {string[]} aliases for unit
         * @throws {QtyError} if unit is unknown
         */
        function getAliases(unitName) {
          if (!UNIT_MAP[unitName]) {
            throw new QtyError("Unit not recognized");
          }
          return UNITS[UNIT_MAP[unitName]][0];
        }
        var SIGNATURE_VECTOR = ["length", "time", "temperature", "mass", "current", "substance", "luminosity", "currency", "information", "angle"];

        /*
        calculates the unit signature id for use in comparing compatible units and simplification
        the signature is based on a simple classification of units and is based on the following publication
         Novak, G.S., Jr. "Conversion of units of measurement", IEEE Transactions on Software Engineering,
        21(8), Aug 1995, pp.651-661
        doi://10.1109/32.403789
        http://ieeexplore.ieee.org/Xplore/login.jsp?url=/iel1/32/9079/00403789.pdf?isnumber=9079&prod=JNL&arnumber=403789&arSt=651&ared=661&arAuthor=Novak%2C+G.S.%2C+Jr.
        */
        function unitSignature() {
          if (this.signature) {
            return this.signature;
          }
          var vector = unitSignatureVector.call(this);
          for (var i = 0; i < vector.length; i++) {
            vector[i] *= Math.pow(20, i);
          }
          return vector.reduce(function (previous, current) {
            return previous + current;
          }, 0);
        }

        // calculates the unit signature vector used by unit_signature
        function unitSignatureVector() {
          if (!this.isBase()) {
            return unitSignatureVector.call(this.toBase());
          }
          var vector = new Array(SIGNATURE_VECTOR.length);
          for (var i = 0; i < vector.length; i++) {
            vector[i] = 0;
          }
          var r, n;
          for (var j = 0; j < this.numerator.length; j++) {
            if (r = UNITS[this.numerator[j]]) {
              n = SIGNATURE_VECTOR.indexOf(r[2]);
              if (n >= 0) {
                vector[n] = vector[n] + 1;
              }
            }
          }
          for (var k = 0; k < this.denominator.length; k++) {
            if (r = UNITS[this.denominator[k]]) {
              n = SIGNATURE_VECTOR.indexOf(r[2]);
              if (n >= 0) {
                vector[n] = vector[n] - 1;
              }
            }
          }
          return vector;
        }
        var SIGN = "[+-]";
        var INTEGER = "\\d+";
        var SIGNED_INTEGER = SIGN + "?" + INTEGER;
        var FRACTION = "\\." + INTEGER;
        var FLOAT = "(?:" + INTEGER + "(?:" + FRACTION + ")?" + ")" + "|" + "(?:" + FRACTION + ")";
        var EXPONENT = "[Ee]" + SIGNED_INTEGER;
        var SCI_NUMBER = "(?:" + FLOAT + ")(?:" + EXPONENT + ")?";
        var SIGNED_NUMBER = SIGN + "?\\s*" + SCI_NUMBER;
        var QTY_STRING = "(" + SIGNED_NUMBER + ")?" + "\\s*([^/]*)(?:\/(.+))?";
        var QTY_STRING_REGEX = new RegExp("^" + QTY_STRING + "$");
        var POWER_OP = "\\^|\\*{2}";
        // Allow unit powers representing scalar, length, area, volume; 4 is for some
        // special case representations in SI base units.
        var SAFE_POWER = "[01234]";
        var TOP_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(-?" + SAFE_POWER + "(?![a-zA-Z]))");
        var BOTTOM_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(" + SAFE_POWER + "(?![a-zA-Z]))");

        /* parse a string into a unit object.
         * Typical formats like :
         * "5.6 kg*m/s^2"
         * "5.6 kg*m*s^-2"
         * "5.6 kilogram*meter*second^-2"
         * "2.2 kPa"
         * "37 degC"
         * "1"  -- creates a unitless constant with value 1
         * "GPa"  -- creates a unit with scalar 1 with units 'GPa'
         * 6'4"  -- recognized as 6 feet + 4 inches
         * 8 lbs 8 oz -- recognized as 8 lbs + 8 ounces
         */
        function parse(val) {
          if (!isString(val)) {
            val = val.toString();
          }
          val = val.trim();
          var result = QTY_STRING_REGEX.exec(val);
          if (!result) {
            throw new QtyError(val + ": Quantity not recognized");
          }
          var scalarMatch = result[1];
          if (scalarMatch) {
            // Allow whitespaces between sign and scalar for loose parsing
            scalarMatch = scalarMatch.replace(/\s/g, "");
            this.scalar = parseFloat(scalarMatch);
          } else {
            this.scalar = 1;
          }
          var top = result[2];
          var bottom = result[3];
          var n, x, nx;
          // TODO DRY me
          while (result = TOP_REGEX.exec(top)) {
            n = parseFloat(result[2]);
            if (isNaN(n)) {
              // Prevents infinite loops
              throw new QtyError("Unit exponent is not a number");
            }
            // Disallow unrecognized unit even if exponent is 0
            if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
              throw new QtyError("Unit not recognized");
            }
            x = result[1] + " ";
            nx = "";
            for (var i = 0; i < Math.abs(n); i++) {
              nx += x;
            }
            if (n >= 0) {
              top = top.replace(result[0], nx);
            } else {
              bottom = bottom ? bottom + nx : nx;
              top = top.replace(result[0], "");
            }
          }
          while (result = BOTTOM_REGEX.exec(bottom)) {
            n = parseFloat(result[2]);
            if (isNaN(n)) {
              // Prevents infinite loops
              throw new QtyError("Unit exponent is not a number");
            }
            // Disallow unrecognized unit even if exponent is 0
            if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
              throw new QtyError("Unit not recognized");
            }
            x = result[1] + " ";
            nx = "";
            for (var j = 0; j < n; j++) {
              nx += x;
            }
            bottom = bottom.replace(result[0], nx);
          }
          if (top) {
            this.numerator = parseUnits(top.trim());
          }
          if (bottom) {
            this.denominator = parseUnits(bottom.trim());
          }
        }
        var PREFIX_REGEX = Object.keys(PREFIX_MAP).sort(function (a, b) {
          return b.length - a.length;
        }).join("|");
        var UNIT_REGEX = Object.keys(UNIT_MAP).sort(function (a, b) {
          return b.length - a.length;
        }).join("|");
        /*
         * Minimal boundary regex to support units with Unicode characters
         * \b only works for ASCII
         */
        var BOUNDARY_REGEX = "\\b|$";
        var UNIT_MATCH = "(" + PREFIX_REGEX + ")??(" + UNIT_REGEX + ")(?:" + BOUNDARY_REGEX + ")";
        var UNIT_TEST_REGEX = new RegExp("^\\s*(" + UNIT_MATCH + "[\\s\\*]*)+$");
        var UNIT_MATCH_REGEX = new RegExp(UNIT_MATCH, "g"); // g flag for multiple occurences
        var parsedUnitsCache = {};
        /**
         * Parses and converts units string to normalized unit array.
         * Result is cached to speed up next calls.
         *
         * @param {string} units Units string
         * @returns {string[]} Array of normalized units
         *
         * @example
         * // Returns ["<second>", "<meter>", "<second>"]
         * parseUnits("s m s");
         *
         */
        function parseUnits(units) {
          var cached = parsedUnitsCache[units];
          if (cached) {
            return cached;
          }
          var unitMatch,
            normalizedUnits = [];

          // Scan
          if (!UNIT_TEST_REGEX.test(units)) {
            throw new QtyError("Unit not recognized");
          }
          while (unitMatch = UNIT_MATCH_REGEX.exec(units)) {
            normalizedUnits.push(unitMatch.slice(1));
          }
          normalizedUnits = normalizedUnits.map(function (item) {
            return PREFIX_MAP[item[0]] ? [PREFIX_MAP[item[0]], UNIT_MAP[item[1]]] : [UNIT_MAP[item[1]]];
          });

          // Flatten and remove null elements
          normalizedUnits = normalizedUnits.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          normalizedUnits = normalizedUnits.filter(function (item) {
            return item;
          });
          parsedUnitsCache[units] = normalizedUnits;
          return normalizedUnits;
        }

        /**
         * Parses a string as a quantity
         * @param {string} value - quantity as text
         * @throws if value is not a string
         * @returns {Qty|null} Parsed quantity or null if unrecognized
         */
        function globalParse(value) {
          if (!isString(value)) {
            throw new QtyError("Argument should be a string");
          }
          try {
            return this(value);
          } catch (e) {
            return null;
          }
        }

        /**
         * Tests if a value is a Qty instance
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a Qty instance, false otherwise
         */
        function isQty(value) {
          return value instanceof Qty;
        }
        function Qty(initValue, initUnits) {
          assertValidConstructorArgs.apply(null, arguments);
          if (!isQty(this)) {
            return new Qty(initValue, initUnits);
          }
          this.scalar = null;
          this.baseScalar = null;
          this.signature = null;
          this._conversionCache = {};
          this.numerator = UNITY_ARRAY;
          this.denominator = UNITY_ARRAY;
          if (isDefinitionObject(initValue)) {
            this.scalar = initValue.scalar;
            this.numerator = initValue.numerator && initValue.numerator.length !== 0 ? initValue.numerator : UNITY_ARRAY;
            this.denominator = initValue.denominator && initValue.denominator.length !== 0 ? initValue.denominator : UNITY_ARRAY;
          } else if (initUnits) {
            parse.call(this, initUnits);
            this.scalar = initValue;
          } else {
            parse.call(this, initValue);
          }

          // math with temperatures is very limited
          if (this.denominator.join("*").indexOf("temp") >= 0) {
            throw new QtyError("Cannot divide with temperatures");
          }
          if (this.numerator.join("*").indexOf("temp") >= 0) {
            if (this.numerator.length > 1) {
              throw new QtyError("Cannot multiply by temperatures");
            }
            if (!compareArray(this.denominator, UNITY_ARRAY)) {
              throw new QtyError("Cannot divide with temperatures");
            }
          }
          this.initValue = initValue;
          updateBaseScalar.call(this);
          if (this.isTemperature() && this.baseScalar < 0) {
            throw new QtyError("Temperatures must not be less than absolute zero");
          }
        }
        Qty.prototype = {
          // Properly set up constructor
          constructor: Qty
        };

        /**
         * Asserts constructor arguments are valid
         *
         * @param {*} value - Value to test
         * @param {string} [units] - Optional units when value is passed as a number
         *
         * @returns {void}
         * @throws {QtyError} if constructor arguments are invalid
         */
        function assertValidConstructorArgs(value, units) {
          if (units) {
            if (!(isNumber(value) && isString(units))) {
              throw new QtyError("Only number accepted as initialization value " + "when units are explicitly provided");
            }
          } else {
            if (!(isString(value) || isNumber(value) || isQty(value) || isDefinitionObject(value))) {
              throw new QtyError("Only string, number or quantity accepted as " + "single initialization value");
            }
          }
        }

        /**
         * Tests if a value is a Qty definition object
         *
         * @param {*} value - Value to test
         *
         * @returns {boolean} true if value is a definition object, false otherwise
         */
        function isDefinitionObject(value) {
          return value && typeof value === "object" && value.hasOwnProperty("scalar");
        }
        function updateBaseScalar() {
          if (this.baseScalar) {
            return this.baseScalar;
          }
          if (this.isBase()) {
            this.baseScalar = this.scalar;
            this.signature = unitSignature.call(this);
          } else {
            var base = this.toBase();
            this.baseScalar = base.scalar;
            this.signature = base.signature;
          }
        }
        var KINDS = {
          "-312078": "elastance",
          "-312058": "resistance",
          "-312038": "inductance",
          "-152058": "potential",
          "-152040": "magnetism",
          "-152038": "magnetism",
          "-7997": "specific_volume",
          "-79": "snap",
          "-59": "jolt",
          "-39": "acceleration",
          "-38": "radiation",
          "-20": "frequency",
          "-19": "speed",
          "-18": "viscosity",
          "-17": "volumetric_flow",
          "-1": "wavenumber",
          "0": "unitless",
          "1": "length",
          "2": "area",
          "3": "volume",
          "20": "time",
          "400": "temperature",
          "7941": "yank",
          "7942": "power",
          "7959": "pressure",
          "7961": "force",
          "7962": "energy",
          "7979": "viscosity",
          "7981": "momentum",
          "7982": "angular_momentum",
          "7997": "density",
          "7998": "area_density",
          "8000": "mass",
          "152020": "radiation_exposure",
          "159999": "magnetism",
          "160000": "current",
          "160020": "charge",
          "312058": "conductance",
          "312078": "capacitance",
          "3199980": "activity",
          "3199997": "molar_concentration",
          "3200000": "substance",
          "63999998": "illuminance",
          "64000000": "luminous_power",
          "1280000000": "currency",
          "25599999980": "information_rate",
          "25600000000": "information",
          "511999999980": "angular_velocity",
          "512000000000": "angle"
        };

        /**
         * Returns the list of available well-known kinds of units, e.g.
         * "radiation" or "length".
         *
         * @returns {string[]} names of kinds of units
         */
        function getKinds() {
          return uniq(Object.keys(KINDS).map(function (knownSignature) {
            return KINDS[knownSignature];
          }));
        }
        Qty.prototype.kind = function () {
          return KINDS[this.signature.toString()];
        };
        assign(Qty.prototype, {
          isDegrees: function () {
            // signature may not have been calculated yet
            return (this.signature === null || this.signature === 400) && this.numerator.length === 1 && compareArray(this.denominator, UNITY_ARRAY) && (this.numerator[0].match(/<temp-[CFRK]>/) || this.numerator[0].match(/<(kelvin|celsius|rankine|fahrenheit)>/));
          },
          isTemperature: function () {
            return this.isDegrees() && this.numerator[0].match(/<temp-[CFRK]>/);
          }
        });
        function subtractTemperatures(lhs, rhs) {
          var lhsUnits = lhs.units();
          var rhsConverted = rhs.to(lhsUnits);
          var dstDegrees = Qty(getDegreeUnits(lhsUnits));
          return Qty({
            "scalar": lhs.scalar - rhsConverted.scalar,
            "numerator": dstDegrees.numerator,
            "denominator": dstDegrees.denominator
          });
        }
        function subtractTempDegrees(temp, deg) {
          var tempDegrees = deg.to(getDegreeUnits(temp.units()));
          return Qty({
            "scalar": temp.scalar - tempDegrees.scalar,
            "numerator": temp.numerator,
            "denominator": temp.denominator
          });
        }
        function addTempDegrees(temp, deg) {
          var tempDegrees = deg.to(getDegreeUnits(temp.units()));
          return Qty({
            "scalar": temp.scalar + tempDegrees.scalar,
            "numerator": temp.numerator,
            "denominator": temp.denominator
          });
        }
        function getDegreeUnits(units) {
          if (units === "tempK") {
            return "degK";
          } else if (units === "tempC") {
            return "degC";
          } else if (units === "tempF") {
            return "degF";
          } else if (units === "tempR") {
            return "degR";
          } else {
            throw new QtyError("Unknown type for temp conversion from: " + units);
          }
        }
        function toDegrees(src, dst) {
          var srcDegK = toDegK(src);
          var dstUnits = dst.units();
          var dstScalar;
          if (dstUnits === "degK") {
            dstScalar = srcDegK.scalar;
          } else if (dstUnits === "degC") {
            dstScalar = srcDegK.scalar;
          } else if (dstUnits === "degF") {
            dstScalar = srcDegK.scalar * 9 / 5;
          } else if (dstUnits === "degR") {
            dstScalar = srcDegK.scalar * 9 / 5;
          } else {
            throw new QtyError("Unknown type for degree conversion to: " + dstUnits);
          }
          return Qty({
            "scalar": dstScalar,
            "numerator": dst.numerator,
            "denominator": dst.denominator
          });
        }
        function toDegK(qty) {
          var units = qty.units();
          var q;
          if (units.match(/(deg)[CFRK]/)) {
            q = qty.baseScalar;
          } else if (units === "tempK") {
            q = qty.scalar;
          } else if (units === "tempC") {
            q = qty.scalar;
          } else if (units === "tempF") {
            q = qty.scalar * 5 / 9;
          } else if (units === "tempR") {
            q = qty.scalar * 5 / 9;
          } else {
            throw new QtyError("Unknown type for temp conversion from: " + units);
          }
          return Qty({
            "scalar": q,
            "numerator": ["<kelvin>"],
            "denominator": UNITY_ARRAY
          });
        }
        function toTemp(src, dst) {
          var dstUnits = dst.units();
          var dstScalar;
          if (dstUnits === "tempK") {
            dstScalar = src.baseScalar;
          } else if (dstUnits === "tempC") {
            dstScalar = src.baseScalar - 273.15;
          } else if (dstUnits === "tempF") {
            dstScalar = src.baseScalar * 9 / 5 - 459.67;
          } else if (dstUnits === "tempR") {
            dstScalar = src.baseScalar * 9 / 5;
          } else {
            throw new QtyError("Unknown type for temp conversion to: " + dstUnits);
          }
          return Qty({
            "scalar": dstScalar,
            "numerator": dst.numerator,
            "denominator": dst.denominator
          });
        }
        function toTempK(qty) {
          var units = qty.units();
          var q;
          if (units.match(/(deg)[CFRK]/)) {
            q = qty.baseScalar;
          } else if (units === "tempK") {
            q = qty.scalar;
          } else if (units === "tempC") {
            q = qty.scalar + 273.15;
          } else if (units === "tempF") {
            q = (qty.scalar + 459.67) * 5 / 9;
          } else if (units === "tempR") {
            q = qty.scalar * 5 / 9;
          } else {
            throw new QtyError("Unknown type for temp conversion from: " + units);
          }
          return Qty({
            "scalar": q,
            "numerator": ["<temp-K>"],
            "denominator": UNITY_ARRAY
          });
        }
        assign(Qty.prototype, {
          /**
           * Converts to other compatible units.
           * Instance's converted quantities are cached for faster subsequent calls.
           *
           * @param {(string|Qty)} other - Target units as string or retrieved from
           *                               other Qty instance (scalar is ignored)
           *
           * @returns {Qty} New converted Qty instance with target units
           *
           * @throws {QtyError} if target units are incompatible
           *
           * @example
           * var weight = Qty("25 kg");
           * weight.to("lb"); // => Qty("55.11556554621939 lbs");
           * weight.to(Qty("3 g")); // => Qty("25000 g"); // scalar of passed Qty is ignored
           */
          to: function (other) {
            var cached, target;
            if (other === undefined || other === null) {
              return this;
            }
            if (!isString(other)) {
              return this.to(other.units());
            }
            cached = this._conversionCache[other];
            if (cached) {
              return cached;
            }

            // Instantiating target to normalize units
            target = Qty(other);
            if (target.units() === this.units()) {
              return this;
            }
            if (!this.isCompatible(target)) {
              if (this.isInverse(target)) {
                target = this.inverse().to(other);
              } else {
                throwIncompatibleUnits(this.units(), target.units());
              }
            } else {
              if (target.isTemperature()) {
                target = toTemp(this, target);
              } else if (target.isDegrees()) {
                target = toDegrees(this, target);
              } else {
                var q = divSafe(this.baseScalar, target.baseScalar);
                target = Qty({
                  "scalar": q,
                  "numerator": target.numerator,
                  "denominator": target.denominator
                });
              }
            }
            this._conversionCache[other] = target;
            return target;
          },
          // convert to base SI units
          // results of the conversion are cached so subsequent calls to this will be fast
          toBase: function () {
            if (this.isBase()) {
              return this;
            }
            if (this.isTemperature()) {
              return toTempK(this);
            }
            var cached = baseUnitCache[this.units()];
            if (!cached) {
              cached = toBaseUnits(this.numerator, this.denominator);
              baseUnitCache[this.units()] = cached;
            }
            return cached.mul(this.scalar);
          },
          // Converts the unit back to a float if it is unitless.  Otherwise raises an exception
          toFloat: function () {
            if (this.isUnitless()) {
              return this.scalar;
            }
            throw new QtyError("Can't convert to Float unless unitless.  Use Unit#scalar");
          },
          /**
           * Returns the nearest multiple of quantity passed as
           * precision
           *
           * @param {(Qty|string|number)} precQuantity - Quantity, string formated
           *   quantity or number as expected precision
           *
           * @returns {Qty} Nearest multiple of precQuantity
           *
           * @example
           * Qty('5.5 ft').toPrec('2 ft'); // returns 6 ft
           * Qty('0.8 cu').toPrec('0.25 cu'); // returns 0.75 cu
           * Qty('6.3782 m').toPrec('cm'); // returns 6.38 m
           * Qty('1.146 MPa').toPrec('0.1 bar'); // returns 1.15 MPa
           *
           */
          toPrec: function (precQuantity) {
            if (isString(precQuantity)) {
              precQuantity = Qty(precQuantity);
            }
            if (isNumber(precQuantity)) {
              precQuantity = Qty(precQuantity + " " + this.units());
            }
            if (!this.isUnitless()) {
              precQuantity = precQuantity.to(this.units());
            } else if (!precQuantity.isUnitless()) {
              throwIncompatibleUnits(this.units(), precQuantity.units());
            }
            if (precQuantity.scalar === 0) {
              throw new QtyError("Divide by zero");
            }
            var precRoundedResult = mulSafe(Math.round(this.scalar / precQuantity.scalar), precQuantity.scalar);
            return Qty(precRoundedResult + this.units());
          }
        });

        /**
         * Configures and returns a fast function to convert
         * Number values from units to others.
         * Useful to efficiently convert large array of values
         * with same units into others with iterative methods.
         * Does not take care of rounding issues.
         *
         * @param {string} srcUnits Units of values to convert
         * @param {string} dstUnits Units to convert to
         *
         * @returns {Function} Converting function accepting Number value
         *   and returning converted value
         *
         * @throws "Incompatible units" if units are incompatible
         *
         * @example
         * // Converting large array of numbers with the same units
         * // into other units
         * var converter = Qty.swiftConverter("m/h", "ft/s");
         * var convertedSerie = largeSerie.map(converter);
         *
         */
        function swiftConverter(srcUnits, dstUnits) {
          var srcQty = Qty(srcUnits);
          var dstQty = Qty(dstUnits);
          if (srcQty.eq(dstQty)) {
            return identity;
          }
          var convert;
          if (!srcQty.isTemperature()) {
            convert = function (value) {
              return value * srcQty.baseScalar / dstQty.baseScalar;
            };
          } else {
            convert = function (value) {
              // TODO Not optimized
              return srcQty.mul(value).to(dstQty).scalar;
            };
          }
          return function converter(value) {
            var i, length, result;
            if (!Array.isArray(value)) {
              return convert(value);
            } else {
              length = value.length;
              result = [];
              for (i = 0; i < length; i++) {
                result.push(convert(value[i]));
              }
              return result;
            }
          };
        }
        var baseUnitCache = {};
        function toBaseUnits(numerator, denominator) {
          var num = [];
          var den = [];
          var q = 1;
          var unit;
          for (var i = 0; i < numerator.length; i++) {
            unit = numerator[i];
            if (PREFIX_VALUES[unit]) {
              // workaround to fix
              // 0.1 * 0.1 => 0.010000000000000002
              q = mulSafe(q, PREFIX_VALUES[unit]);
            } else {
              if (UNIT_VALUES[unit]) {
                q *= UNIT_VALUES[unit].scalar;
                if (UNIT_VALUES[unit].numerator) {
                  num.push(UNIT_VALUES[unit].numerator);
                }
                if (UNIT_VALUES[unit].denominator) {
                  den.push(UNIT_VALUES[unit].denominator);
                }
              }
            }
          }
          for (var j = 0; j < denominator.length; j++) {
            unit = denominator[j];
            if (PREFIX_VALUES[unit]) {
              q /= PREFIX_VALUES[unit];
            } else {
              if (UNIT_VALUES[unit]) {
                q /= UNIT_VALUES[unit].scalar;
                if (UNIT_VALUES[unit].numerator) {
                  den.push(UNIT_VALUES[unit].numerator);
                }
                if (UNIT_VALUES[unit].denominator) {
                  num.push(UNIT_VALUES[unit].denominator);
                }
              }
            }
          }

          // Flatten
          num = num.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          den = den.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          return Qty({
            "scalar": q,
            "numerator": num,
            "denominator": den
          });
        }
        Qty.parse = globalParse;
        Qty.getUnits = getUnits;
        Qty.getAliases = getAliases;
        Qty.mulSafe = mulSafe;
        Qty.divSafe = divSafe;
        Qty.getKinds = getKinds;
        Qty.swiftConverter = swiftConverter;
        Qty.Error = QtyError;
        assign(Qty.prototype, {
          // Returns new instance with units of this
          add: function (other) {
            if (isString(other)) {
              other = Qty(other);
            }
            if (!this.isCompatible(other)) {
              throwIncompatibleUnits(this.units(), other.units());
            }
            if (this.isTemperature() && other.isTemperature()) {
              throw new QtyError("Cannot add two temperatures");
            } else if (this.isTemperature()) {
              return addTempDegrees(this, other);
            } else if (other.isTemperature()) {
              return addTempDegrees(other, this);
            }
            return Qty({
              "scalar": this.scalar + other.to(this).scalar,
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          },
          sub: function (other) {
            if (isString(other)) {
              other = Qty(other);
            }
            if (!this.isCompatible(other)) {
              throwIncompatibleUnits(this.units(), other.units());
            }
            if (this.isTemperature() && other.isTemperature()) {
              return subtractTemperatures(this, other);
            } else if (this.isTemperature()) {
              return subtractTempDegrees(this, other);
            } else if (other.isTemperature()) {
              throw new QtyError("Cannot subtract a temperature from a differential degree unit");
            }
            return Qty({
              "scalar": this.scalar - other.to(this).scalar,
              "numerator": this.numerator,
              "denominator": this.denominator
            });
          },
          mul: function (other) {
            if (isNumber(other)) {
              return Qty({
                "scalar": mulSafe(this.scalar, other),
                "numerator": this.numerator,
                "denominator": this.denominator
              });
            } else if (isString(other)) {
              other = Qty(other);
            }
            if ((this.isTemperature() || other.isTemperature()) && !(this.isUnitless() || other.isUnitless())) {
              throw new QtyError("Cannot multiply by temperatures");
            }

            // Quantities should be multiplied with same units if compatible, with base units else
            var op1 = this;
            var op2 = other;

            // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
            // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
            if (op1.isCompatible(op2) && op1.signature !== 400) {
              op2 = op2.to(op1);
            }
            var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.numerator, op2.denominator);
            return Qty({
              "scalar": mulSafe(op1.scalar, op2.scalar, numdenscale[2]),
              "numerator": numdenscale[0],
              "denominator": numdenscale[1]
            });
          },
          div: function (other) {
            if (isNumber(other)) {
              if (other === 0) {
                throw new QtyError("Divide by zero");
              }
              return Qty({
                "scalar": this.scalar / other,
                "numerator": this.numerator,
                "denominator": this.denominator
              });
            } else if (isString(other)) {
              other = Qty(other);
            }
            if (other.scalar === 0) {
              throw new QtyError("Divide by zero");
            }
            if (other.isTemperature()) {
              throw new QtyError("Cannot divide with temperatures");
            } else if (this.isTemperature() && !other.isUnitless()) {
              throw new QtyError("Cannot divide with temperatures");
            }

            // Quantities should be multiplied with same units if compatible, with base units else
            var op1 = this;
            var op2 = other;

            // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
            // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
            if (op1.isCompatible(op2) && op1.signature !== 400) {
              op2 = op2.to(op1);
            }
            var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.denominator, op2.numerator);
            return Qty({
              "scalar": mulSafe(op1.scalar, numdenscale[2]) / op2.scalar,
              "numerator": numdenscale[0],
              "denominator": numdenscale[1]
            });
          },
          // Returns a Qty that is the inverse of this Qty,
          inverse: function () {
            if (this.isTemperature()) {
              throw new QtyError("Cannot divide with temperatures");
            }
            if (this.scalar === 0) {
              throw new QtyError("Divide by zero");
            }
            return Qty({
              "scalar": 1 / this.scalar,
              "numerator": this.denominator,
              "denominator": this.numerator
            });
          }
        });
        function cleanTerms(num1, den1, num2, den2) {
          function notUnity(val) {
            return val !== UNITY;
          }
          num1 = num1.filter(notUnity);
          num2 = num2.filter(notUnity);
          den1 = den1.filter(notUnity);
          den2 = den2.filter(notUnity);
          var combined = {};
          function combineTerms(terms, direction) {
            var k;
            var prefix;
            var prefixValue;
            for (var i = 0; i < terms.length; i++) {
              if (PREFIX_VALUES[terms[i]]) {
                k = terms[i + 1];
                prefix = terms[i];
                prefixValue = PREFIX_VALUES[prefix];
                i++;
              } else {
                k = terms[i];
                prefix = null;
                prefixValue = 1;
              }
              if (k && k !== UNITY) {
                if (combined[k]) {
                  combined[k][0] += direction;
                  var combinedPrefixValue = combined[k][2] ? PREFIX_VALUES[combined[k][2]] : 1;
                  combined[k][direction === 1 ? 3 : 4] *= divSafe(prefixValue, combinedPrefixValue);
                } else {
                  combined[k] = [direction, k, prefix, 1, 1];
                }
              }
            }
          }
          combineTerms(num1, 1);
          combineTerms(den1, -1);
          combineTerms(num2, 1);
          combineTerms(den2, -1);
          var num = [];
          var den = [];
          var scale = 1;
          for (var prop in combined) {
            if (combined.hasOwnProperty(prop)) {
              var item = combined[prop];
              var n;
              if (item[0] > 0) {
                for (n = 0; n < item[0]; n++) {
                  num.push(item[2] === null ? item[1] : [item[2], item[1]]);
                }
              } else if (item[0] < 0) {
                for (n = 0; n < -item[0]; n++) {
                  den.push(item[2] === null ? item[1] : [item[2], item[1]]);
                }
              }
              scale *= divSafe(item[3], item[4]);
            }
          }
          if (num.length === 0) {
            num = UNITY_ARRAY;
          }
          if (den.length === 0) {
            den = UNITY_ARRAY;
          }

          // Flatten
          num = num.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          den = den.reduce(function (a, b) {
            return a.concat(b);
          }, []);
          return [num, den, scale];
        }
        assign(Qty.prototype, {
          eq: function (other) {
            return this.compareTo(other) === 0;
          },
          lt: function (other) {
            return this.compareTo(other) === -1;
          },
          lte: function (other) {
            return this.eq(other) || this.lt(other);
          },
          gt: function (other) {
            return this.compareTo(other) === 1;
          },
          gte: function (other) {
            return this.eq(other) || this.gt(other);
          },
          // Compare two Qty objects. Throws an exception if they are not of compatible types.
          // Comparisons are done based on the value of the quantity in base SI units.
          //
          // NOTE: We cannot compare inverses as that breaks the general compareTo contract:
          //   if a.compareTo(b) < 0 then b.compareTo(a) > 0
          //   if a.compareTo(b) == 0 then b.compareTo(a) == 0
          //
          //   Since "10S" == ".1ohm" (10 > .1) and "10ohm" == ".1S" (10 > .1)
          //     Qty("10S").inverse().compareTo("10ohm") == -1
          //     Qty("10ohm").inverse().compareTo("10S") == -1
          //
          //   If including inverses in the sort is needed, I suggest writing: Qty.sort(qtyArray,units)
          compareTo: function (other) {
            if (isString(other)) {
              return this.compareTo(Qty(other));
            }
            if (!this.isCompatible(other)) {
              throwIncompatibleUnits(this.units(), other.units());
            }
            if (this.baseScalar < other.baseScalar) {
              return -1;
            } else if (this.baseScalar === other.baseScalar) {
              return 0;
            } else if (this.baseScalar > other.baseScalar) {
              return 1;
            }
          },
          // Return true if quantities and units match
          // Unit("100 cm").same(Unit("100 cm"))  # => true
          // Unit("100 cm").same(Unit("1 m"))     # => false
          same: function (other) {
            return this.scalar === other.scalar && this.units() === other.units();
          }
        });
        assign(Qty.prototype, {
          // returns true if no associated units
          // false, even if the units are "unitless" like 'radians, each, etc'
          isUnitless: function () {
            return [this.numerator, this.denominator].every(function (item) {
              return compareArray(item, UNITY_ARRAY);
            });
          },
          /*
          check to see if units are compatible, but not the scalar part
          this check is done by comparing signatures for performance reasons
          if passed a string, it will create a unit object with the string and then do the comparison
          this permits a syntax like:
          unit =~ "mm"
          if you want to do a regexp on the unit string do this ...
          unit.units =~ /regexp/
          */
          isCompatible: function (other) {
            if (isString(other)) {
              return this.isCompatible(Qty(other));
            }
            if (!isQty(other)) {
              return false;
            }
            if (other.signature !== undefined) {
              return this.signature === other.signature;
            } else {
              return false;
            }
          },
          /*
          check to see if units are inverse of each other, but not the scalar part
          this check is done by comparing signatures for performance reasons
          if passed a string, it will create a unit object with the string and then do the comparison
          this permits a syntax like:
          unit =~ "mm"
          if you want to do a regexp on the unit string do this ...
          unit.units =~ /regexp/
          */
          isInverse: function (other) {
            return this.inverse().isCompatible(other);
          },
          // Returns 'true' if the Unit is represented in base units
          isBase: function () {
            if (this._isBase !== undefined) {
              return this._isBase;
            }
            if (this.isDegrees() && this.numerator[0].match(/<(kelvin|temp-K)>/)) {
              this._isBase = true;
              return this._isBase;
            }
            this.numerator.concat(this.denominator).forEach(function (item) {
              if (item !== UNITY && BASE_UNITS.indexOf(item) === -1) {
                this._isBase = false;
              }
            }, this);
            if (this._isBase === false) {
              return this._isBase;
            }
            this._isBase = true;
            return this._isBase;
          }
        });
        function NestedMap() {}
        NestedMap.prototype.get = function (keys) {
          // Allows to pass key1, key2, ... instead of [key1, key2, ...]
          if (arguments.length > 1) {
            // Slower with Firefox but faster with Chrome than
            // Array.prototype.slice.call(arguments)
            // See http://jsperf.com/array-apply-versus-array-prototype-slice-call
            keys = Array.apply(null, arguments);
          }
          return keys.reduce(function (map, key, index) {
            if (map) {
              var childMap = map[key];
              if (index === keys.length - 1) {
                return childMap ? childMap.data : undefined;
              } else {
                return childMap;
              }
            }
          }, this);
        };
        NestedMap.prototype.set = function (keys, value) {
          if (arguments.length > 2) {
            keys = Array.prototype.slice.call(arguments, 0, -1);
            value = arguments[arguments.length - 1];
          }
          return keys.reduce(function (map, key, index) {
            var childMap = map[key];
            if (childMap === undefined) {
              childMap = map[key] = {};
            }
            if (index === keys.length - 1) {
              childMap.data = value;
              return value;
            } else {
              return childMap;
            }
          }, this);
        };

        /**
         * Default formatter
         *
         * @param {number} scalar - scalar value
         * @param {string} units - units as string
         *
         * @returns {string} formatted result
         */
        function defaultFormatter(scalar, units) {
          return (scalar + " " + units).trim();
        }

        /**
         *
         * Configurable Qty default formatter
         *
         * @type {function}
         *
         * @param {number} scalar
         * @param {string} units
         *
         * @returns {string} formatted result
         */
        Qty.formatter = defaultFormatter;
        assign(Qty.prototype, {
          // returns the 'unit' part of the Unit object without the scalar
          units: function () {
            if (this._units !== undefined) {
              return this._units;
            }
            var numIsUnity = compareArray(this.numerator, UNITY_ARRAY);
            var denIsUnity = compareArray(this.denominator, UNITY_ARRAY);
            if (numIsUnity && denIsUnity) {
              this._units = "";
              return this._units;
            }
            var numUnits = stringifyUnits(this.numerator);
            var denUnits = stringifyUnits(this.denominator);
            this._units = numUnits + (denIsUnity ? "" : "/" + denUnits);
            return this._units;
          },
          /**
           * Stringifies the quantity
           * Deprecation notice: only units parameter is supported.
           *
           * @param {(number|string|Qty)} targetUnitsOrMaxDecimalsOrPrec -
           *                              target units if string,
           *                              max number of decimals if number,
           *                              passed to #toPrec before converting if Qty
           *
           * @param {number=} maxDecimals - Maximum number of decimals of
           *                                formatted output
           *
           * @returns {string} reparseable quantity as string
           */
          toString: function (targetUnitsOrMaxDecimalsOrPrec, maxDecimals) {
            var targetUnits;
            if (isNumber(targetUnitsOrMaxDecimalsOrPrec)) {
              targetUnits = this.units();
              maxDecimals = targetUnitsOrMaxDecimalsOrPrec;
            } else if (isString(targetUnitsOrMaxDecimalsOrPrec)) {
              targetUnits = targetUnitsOrMaxDecimalsOrPrec;
            } else if (isQty(targetUnitsOrMaxDecimalsOrPrec)) {
              return this.toPrec(targetUnitsOrMaxDecimalsOrPrec).toString(maxDecimals);
            }
            var out = this.to(targetUnits);
            var outScalar = maxDecimals !== undefined ? round(out.scalar, maxDecimals) : out.scalar;
            out = (outScalar + " " + out.units()).trim();
            return out;
          },
          /**
           * Format the quantity according to optional passed target units
           * and formatter
           *
           * @param {string} [targetUnits=current units] -
           *                 optional units to convert to before formatting
           *
           * @param {function} [formatter=Qty.formatter] -
           *                   delegates formatting to formatter callback.
           *                   formatter is called back with two parameters (scalar, units)
           *                   and should return formatted result.
           *                   If unspecified, formatting is delegated to default formatter
           *                   set to Qty.formatter
           *
           * @example
           * var roundingAndLocalizingFormatter = function(scalar, units) {
           *   // localize or limit scalar to n max decimals for instance
           *   // return formatted result
           * };
           * var qty = Qty('1.1234 m');
           * qty.format(); // same units, default formatter => "1.234 m"
           * qty.format("cm"); // converted to "cm", default formatter => "123.45 cm"
           * qty.format(roundingAndLocalizingFormatter); // same units, custom formatter => "1,2 m"
           * qty.format("cm", roundingAndLocalizingFormatter); // convert to "cm", custom formatter => "123,4 cm"
           *
           * @returns {string} quantity as string
           */
          format: function (targetUnits, formatter) {
            if (arguments.length === 1) {
              if (typeof targetUnits === "function") {
                formatter = targetUnits;
                targetUnits = undefined;
              }
            }
            formatter = formatter || Qty.formatter;
            var targetQty = this.to(targetUnits);
            return formatter.call(this, targetQty.scalar, targetQty.units());
          }
        });
        var stringifiedUnitsCache = new NestedMap();
        /**
         * Returns a string representing a normalized unit array
         *
         * @param {string[]} units Normalized unit array
         * @returns {string} String representing passed normalized unit array and
         *   suitable for output
         *
         */
        function stringifyUnits(units) {
          var stringified = stringifiedUnitsCache.get(units);
          if (stringified) {
            return stringified;
          }
          var isUnity = compareArray(units, UNITY_ARRAY);
          if (isUnity) {
            stringified = "1";
          } else {
            stringified = simplify(getOutputNames(units)).join("*");
          }

          // Cache result
          stringifiedUnitsCache.set(units, stringified);
          return stringified;
        }
        function getOutputNames(units) {
          var unitNames = [],
            token,
            tokenNext;
          for (var i = 0; i < units.length; i++) {
            token = units[i];
            tokenNext = units[i + 1];
            if (PREFIX_VALUES[token]) {
              unitNames.push(OUTPUT_MAP[token] + OUTPUT_MAP[tokenNext]);
              i++;
            } else {
              unitNames.push(OUTPUT_MAP[token]);
            }
          }
          return unitNames;
        }
        function simplify(units) {
          // this turns ['s','m','s'] into ['s2','m']

          var unitCounts = units.reduce(function (acc, unit) {
            var unitCounter = acc[unit];
            if (!unitCounter) {
              acc.push(unitCounter = acc[unit] = [unit, 0]);
            }
            unitCounter[1]++;
            return acc;
          }, []);
          return unitCounts.map(function (unitCount) {
            return unitCount[0] + (unitCount[1] > 1 ? unitCount[1] : "");
          });
        }
        Qty.version = "1.8.0";
        return Qty;
      });
    })(quantities$1);
    var quantitiesExports = quantities$1.exports;
    var Qty = /*@__PURE__*/getDefaultExportFromCjs(quantitiesExports);

    /**
     * Parse a string that may contain min / max and units values
     * @param {string} string
     * @param {object} [options={}]
     * @param {number} [options.defaultValue]
     * @param {string} [options.defjaultUnits]
     * @param {string} [options.targetUnits]
     * @param {string} [options.optional=false] Is this value optional. Would through an error if empty and not optional
     */
    function parseNumbersUnits(string = '', options = {}) {
      const {
        defaultUnits = undefined,
        targetUnits = undefined,
        defaultValue = undefined,
        optional = false
      } = options;
      let normalized = String(string).replace(/ /g, '').replace(/,/g, '.');
      normalized = normalized.replace(/Â°/g, '°');
      if (normalized.match(/^\(.*\)$/)) {
        normalized = normalized.substring(1, normalized.length - 2);
      }

      // still parenthesis ??? We remove what is after
      normalized = normalized.replace(/\(.*/, '');
      if (optional && !normalized) return {};
      let splitter = /^([0-9.Ee-]*)([a-zA-Z^/ °0-9]*)$/;
      if (!normalized.match(splitter)) {
        throw Error(`Can not parseNumbersUnits of: ${string}`);
      }
      let numbers = normalized.replace(splitter, '$1');
      let units = normalized.replace(splitter, '$2').trim();
      let low;
      let high;
      if (numbers === '') {
        if (defaultValue) {
          numbers = String(defaultValue).replace(/ /g, '');
        } else {
          throw Error(`Can not parseNumbersUnits of: ${string}`);
        }
      }
      if (numbers.includes('--')) {
        // -5--4
        low = Number(numbers.split('--')[0]);
        high = Number(`-${numbers.split('--')[1]}`);
      } else if (numbers.startsWith('-')) {
        // -5 or -1-1 (minus 1 to plus 1)
        let subparts = numbers.split('-');
        switch (subparts.length) {
          case 2:
            low = Number(numbers);
            break;
          case 3:
            low = Number(`-${subparts[1]}`);
            high = Number(+subparts[2]);
            break;
          default:
            throw new Error(`Can not parse: ${string}`);
        }
      } else {
        let subparts = numbers.split('-');
        low = subparts[0] !== '' ? Number(subparts[0]) : undefined;
        high = subparts[1] !== undefined ? Number(subparts[1]) : undefined;
      }
      if (!units) units = defaultUnits;

      // Should we convert the units
      if (targetUnits) {
        let convert = Qty.swiftConverter(units.replace(/°/g, 'temp'), targetUnits.replace(/°/g, 'temp'));
        if (low !== undefined) {
          low = convert(low);
        }
        if (high !== undefined) {
          high = convert(high);
        }
        units = targetUnits;
      }
      return {
        low,
        high,
        units
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.temperature={}]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     * @param {object} [options.pressure={}]
     * @param {number} [options.pressure.defaultValue]
     * @param {string} [options.pressure.defaultUnits]
     * @param {string} [options.pressure.targetUnits]
     * @param {string} [options.pressure.optional]
     */
    function parseBoilingPoint(string, options = {}) {
      let parts = String(string).split(/[@/]|, | at /);
      return {
        temperature: parseNumbersUnits(parts[0], options.temperature),
        pressure: parseNumbersUnits(parts[1], options.pressure)
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.value={}]
     * @param {string} [options.value.defaultUnits]
     * @param {string} [options.value.targetUnits]
     * @param {object} [options.temperature={}]
     * @param {number} [options.temperature.defaultValue]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     */
    function parseDensity(string, options = {}) {
      string = String(string);
      string = string.replace(/\(.*/, '');
      string = string.replace('g/cu cm', 'g/cm^3');
      let parts = String(string).split(/[@]|, | at /);
      let value = parts[0];
      let temperature = parts[1];
      if (temperature) {
        temperature = temperature.replace(/ *t= */, '');
      }
      return {
        value: parseNumbersUnits(value, options.value),
        temperature: parseNumbersUnits(temperature, options.temperature)
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.value={}]
     * @param {string} [options.value.defaultUnits]
     * @param {string} [options.value.targetUnits]
     * @param {object} [options.temperature={}]
     * @param {number} [options.temperature.defaultValue]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     */
    function parseRefractiveIndex(string, options = {}) {
      string = String(string).replace(/^[^0-9.]*/, '');
      let parts = string.split(/at|[@/]|, /);
      let value = parts[0];
      let temperature = parts[1];
      if (temperature) {
        temperature = temperature.replace(/ *t= */, '');
      }
      return {
        value: parseNumbersUnits(value, options.value),
        temperature: parseNumbersUnits(temperature, options.temperature)
      };
    }

    /**
     *
     * @param {String} string
     * @param {object} [options={}]
     * @param {object} [options.temperature={}]
     * @param {string} [options.temperature.defaultUnits]
     * @param {string} [options.temperature.targetUnits]
     * @param {string} [options.pressure.optional]
     * @param {object} [options.pressure={}]
     * @param {string} [options.pressure.defaultUnits]
     * @param {string} [options.pressure.targetUnits]

     */
    function parseVaporPressure(string, options = {}) {
      let parts = String(string).split(/[@/]|, | at /);
      return {
        pressure: parseNumbersUnits(parts[0], options.pressure),
        temperature: parseNumbersUnits(parts[1], options.temperature)
      };
    }

    /**
     * Returns all references cited in the PubChem entry
     *
     * @param data Data of a compound data request to the PubChem API
     * @returns References
     */
    function getReferences(data) {
      const references = jp.query(data, '$.Reference[*]').reduce((ref, entry) => {
        ref[entry.ReferenceNumber] = {
          url: entry.URL,
          sourceName: entry.SourceName,
          name: entry.Name,
          description: entry.Description
        };
        return ref;
      }, {});
      return references;
    }

    function extractExperimentalData(data, sectionName, options = {}) {
      const {
        parser
      } = options;
      const references = getReferences(data);
      const experimentalSection = getExperimentalDataSection(data);
      const results = [];
      if (!experimentalSection || experimentalSection.length === 0) {
        return results;
      }
      jp.query(experimentalSection[0], `$.Section[?(@.TOCHeading==="${sectionName}")].Information[*]`).forEach(entry => {
        jp.query(entry, '$.Value.StringWithMarkup[*].String').forEach(original => {
          const result = {
            reference: references[entry.ReferenceNumber],
            data: {
              original
            }
          };
          if (parser) {
            try {
              result.data.parsed = parser(original, options);
            } catch (error) {
              result.data.parsed = {
                error: error.toString()
              };
            }
          }
          results.push(result);
        });
      });
      return results;
    }
    function getExperimentalDataSection(data) {
      const experimentalData = jp.query(data, '$.Section[?(@.TOCHeading==="Chemical and Physical Properties")]' + '.Section[?(@.TOCHeading==="Experimental Properties")]');
      return experimentalData;
    }

    /**
     * Returns the boiling point of a compound
     *
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     * @returns ExperimentalData
     */
    function getBoilingPoint(data, options = {}) {
      const boilingPoint = extractExperimentalData(data, 'Boiling Point', {
        parser: value => parseBoilingPoint(value, merge({
          pressure: {
            defaultValue: 760,
            defaultUnits: 'torr'
          }
        }, options))
      });
      return boilingPoint;
    }

    /**
     * Returns the refractive index of a compound
     * @param data Data of a compound data request to the PubChem API
     * @returns ExperimentalData
     */
    function getDensity(data = {}, options = {}) {
      const density = extractExperimentalData(data, 'Density', {
        parser: value => {
          return parseDensity(value, options);
        }
      });
      return density;
    }

    /**
     *
     *
     * @export
     * @param {Object} data response of a compound data request to the PubChem API
     * @param {object} [options={}]
     * @param {object} [options.targetUnits]
     * @returns {ExperimentalData}
     */
    function getFlashPoint(data, options = {}) {
      const flashPoint = extractExperimentalData(data, 'Flash Point', {
        parser: value => parseNumbersUnits(value, options.temperature)
      });
      return flashPoint;
    }

    /**
     * Returns the melting point of a compound
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     * @returns ExperimentalData
     */
    function getMeltingPoint(data, options = {}) {
      const meltingPoint = extractExperimentalData(data, 'Melting Point', {
        parser: value => parseNumbersUnits(value, options.temperature)
      });
      return meltingPoint;
    }

    /**
     * Returns the refractive index of a compound
     * @param data Data of a compound data request to the PubChem API
     * @returns ExperimentalData
     */
    function getRefractiveIndex(data = {}) {
      const refractiveIndex = extractExperimentalData(data, 'Refractive Index', {
        parser: value => parseRefractiveIndex(value)
      });
      return refractiveIndex;
    }

    /**
     * Returns the solubility of a compound
     * @param data Data of a compound data request to the PubChem API
     * @returns ExperimentalData
     */
    function getSolubility(data) {
      const solubility = extractExperimentalData(data, 'Solubility');
      return solubility;
    }

    /**
     * Returns the vapor pressure of a compound
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     */
    function getVaporPressure(data, options = {}) {
      const vaporPressure = extractExperimentalData(data, 'Vapor Pressure', {
        parser: value => parseVaporPressure(value, merge({
          temperature: {
            optional: true
          }
        }, options))
      });
      return vaporPressure;
    }

    /**
     * Returns the experimental data of a compound
     * @param data Data of a compound data request to the PubChem API
     * @param options Options for the compound
     * @returns ExperimentalData
     */
    function getExperimentalData(data, options) {
      return {
        boilingPoint: getBoilingPoint(data, options),
        density: getDensity(data, options),
        flashPoint: getFlashPoint(data, options),
        meltingPoint: getMeltingPoint(data, options),
        solubility: getSolubility(data),
        vaporPressure: getVaporPressure(data, options),
        refractiveIndex: getRefractiveIndex(data)
      };
    }

    function getFormula(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Molecular Formula")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'Molecular Formula',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Molecular Formula")]')[0].Description
      };
    }

    // Names and Identifiers
    function getInChI(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChI")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'InChI',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChI")]')[0].Description
      };
    }

    // Names and Identifiers
    function getInChIKey(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChIKey")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'InChIKey',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChIKey")]')[0].Description
      };
    }

    // Names and Identifiers
    function getSMILES(data) {
      return {
        value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="SMILES")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
        label: 'SMILES',
        description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="SMILES")]')[0].Description
      };
    }

    function getIdentifiers(data) {
      return {
        formula: getFormula(data),
        inchi: getInChI(data),
        inchiKey: getInChIKey(data),
        smiles: getSMILES(data)
      };
    }

    const ghsPictogramText = {
      GHS01: 'Explosive',
      GHS02: 'Flammable',
      GHS03: 'Oxidizing',
      GHS04: 'Compressed Gas',
      GHS05: 'Corrosive',
      GHS06: 'Toxic',
      GHS07: 'Harmful Irritant',
      GHS08: 'Health Hazard',
      GHS09: 'Environmental Hazard'
    };

    const hazardStatements = {
      H200: '(Deleted) Unstable Explosive',
      H201: '(Deleted) Explosive; mass explosion hazard',
      H202: '(Deleted) Explosive; severe projection hazard',
      H203: '(Deleted) Explosive; fire, blast or projection hazard',
      H204: 'Fire or projection hazard',
      H205: '(Deleted) May mass explode in fire',
      H206: 'Fire, blast or projection hazard; increased risk of explosion if desensitizing agent is reduced',
      H207: 'Fire or projection hazard; increased risk of explosion if desensitizing agent is reduced',
      H208: 'Fire hazard; increased risk of explosion if desensitizing agent is reduced',
      H209: 'Explosive',
      H210: 'Very sensitive',
      H211: 'Maybe sensitive',
      H220: 'Extremely flammable gas',
      H221: 'Flammable gas',
      H222: 'Extremely flammable aerosol',
      H223: 'Flammable aerosol',
      H224: 'Extremely flammable liquid and vapor',
      H225: 'Highly Flammable liquid and vapor',
      H226: 'Flammable liquid and vapor',
      H227: 'Combustible liquid',
      H228: 'Flammable solid',
      H229: 'Pressurized container: may burst if heated',
      H230: 'May react explosively even in the absence of air',
      H231: 'May react explosively even in the absence of air at elevated pressure and/or temperature',
      H232: 'May ignite spontaneously if exposed to air',
      H240: 'Heating may cause an explosion',
      H241: 'Heating may cause a fire or explosion',
      H242: 'Heating may cause a fire',
      H250: 'Catches fire spontaneously if exposed to air',
      H251: 'Self-heating; may catch fire',
      H252: 'Self-heating in large quantities; may catch fire',
      H260: 'In contact with water releases flammable gases which may ignite spontaneously',
      H261: 'In contact with water releases flammable gas',
      H270: 'May cause or intensify fire; oxidizer',
      H271: 'May cause fire or explosion; strong Oxidizer',
      H272: 'May intensify fire; oxidizer',
      H280: 'Contains gas under pressure; may explode if heated',
      H281: 'Contains refrigerated gas; may cause cryogenic burns or injury',
      H282: 'Extremely flammable chemical under pressure: may explode if heated',
      H283: 'Flammable chemical under pressure: may explode if heated',
      H284: 'Chemical under pressure: may explode if heated',
      H290: 'May be corrosive to metals',
      H300: 'Fatal if swallowed',
      H301: 'Toxic if swallowed',
      H302: 'Harmful if swallowed',
      H303: 'May be harmful if swallowed',
      H304: 'May be fatal if swallowed and enters airways',
      H305: 'May be harmful if swallowed and enters airways',
      H310: 'Fatal in contact with skin',
      H311: 'Toxic in contact with skin',
      H312: 'Harmful in contact with skin',
      H313: 'May be harmful in contact with skin',
      H314: 'Causes severe skin burns and eye damage',
      H315: 'Causes skin irritation',
      H316: 'Causes mild skin irritation',
      H317: 'May cause an allergic skin reaction',
      H318: 'Causes serious eye damage',
      H319: 'Causes serious eye irritation',
      H320: 'Causes eye irritation',
      H330: 'Fatal if inhaled',
      H331: 'Toxic if inhaled',
      H332: 'Harmful if inhaled',
      H333: 'May be harmful if inhaled',
      H334: 'May cause allergy or asthma symptoms or breathing difficulties if inhaled',
      H335: 'May cause respiratory irritation',
      H336: 'May cause drowsiness or dizziness',
      H340: 'May cause genetic defects',
      H341: 'Suspected of causing genetic defects',
      H350: 'May cause cancer',
      H350i: 'May cause cancer by inhalation',
      H351: 'Suspected of causing cancer',
      H360: 'May damage fertility or the unborn child',
      H360F: 'May damage fertility',
      H360D: 'May damage the unborn child',
      H360FD: 'May damage fertility; May damage the unborn child',
      H360Fd: 'May damage fertility; Suspected of damaging the unborn child',
      H360Df: 'May damage the unborn child; Suspected of damaging fertility',
      H361: 'Suspected of damaging fertility or the unborn child',
      H361f: 'Suspected of damaging fertility',
      H361d: 'Suspected of damaging the unborn child',
      H361fd: 'Suspected of damaging fertility; Suspected of damaging the unborn child',
      H362: 'May cause harm to breast-fed children',
      H370: 'Causes damage to organs',
      H371: 'May cause damage to organs',
      H372: 'Causes damage to organs through prolonged or repeated exposure',
      H373: 'May causes damage to organs through prolonged or repeated exposure',
      H400: 'Very toxic to aquatic life',
      H401: 'Toxic to aquatic life',
      H402: 'Harmful to aquatic life',
      H410: 'Very toxic to aquatic life with long lasting effects',
      H411: 'Toxic to aquatic life with long lasting effects',
      H412: 'Harmful to aquatic life with long lasting effects',
      H413: 'May cause long lasting harmful effects to aquatic life',
      H420: 'Harms public health and the environment by destroying ozone in the upper atmosphere',
      'H300+H310': 'Fatal if swallowed or in contact with skin',
      'H300+H330': 'Fatal if swallowed or if inhaled',
      'H310+H330': 'Fatal in contact with skin or if inhaled',
      'H300+H310+H330': 'Fatal if swallowed, in contact with skin or if inhaled',
      'H301+H311': 'Toxic if swallowed or in contact with skin',
      'H301+H331': 'Toxic if swallowed or if inhaled',
      'H311+H331': 'Toxic in contact with skin or if inhaled.',
      'H301+H311+H331': 'Toxic if swallowed, in contact with skin or if inhaled',
      'H302+H312': 'Harmful if swallowed or in contact with skin',
      'H302+H332': 'Harmful if swallowed or if inhaled',
      'H312+H332': 'Harmful in contact with skin or if inhaled',
      'H302+H312+H332': 'Harmful if swallowed, in contact with skin or if inhaled',
      'H303+H313': 'May be harmful if swallowed or in contact with skin',
      'H303+H333': 'May be harmful if swallowed or if inhaled',
      'H313+H333': 'May be harmful in contact with skin or if inhaled',
      'H303+H313+H333': 'May be harmful if swallowed, in contact with skin or if inhaled',
      'H315+H320': 'Cause skin and eye irritation',
      EUH001: 'Explosive when dry',
      EUH006: 'Explosive with or without contact with air',
      EUH014: 'Reacts violently with water',
      EUH018: 'In use may form flammable/explosive vapour-air mixture',
      EUH019: 'May form explosive peroxides',
      EUH044: 'Risk of explosion if heated under confinement',
      EUH029: 'Contact with water liberates toxic gas',
      EUH031: 'Contact with acids liberates toxic gas',
      EUH032: 'Contact with acids liberates very toxic gas',
      EUH066: 'Repeated exposure may cause skin dryness or cracking',
      EUH070: 'Toxic by eye contact',
      EUH071: 'Corrosive to the respiratory tract',
      EUH059: 'Hazardous to the ozone layer, superseded by GHS Class 5.1 in the second adaptation to technical progress of CLP.',
      EUH201: 'Contains lead. Should not be used on surfaces liable to be chewed or sucked by children.',
      EUH201A: 'Warning! Contains lead.',
      EUH202: 'Cyanoacrylate. Danger. Bonds skin and eyes in seconds. Keep out of the reach of children.',
      EUH203: 'Contains chromium(VI). May produce an allergic reaction.',
      EUH204: 'Contains isocyanates. May produce an allergic reaction.',
      EUH205: 'Contains epoxy constituents. May produce an allergic reaction.',
      EUH206: 'Warning! Do not use together with other products. May release dangerous gases (chlorine).',
      EUH207: 'Warning! Contains cadmium. Dangerous fumes are formed during use. See information supplied by the manufacturer. Comply with the safety instructions.',
      EUH208: 'Contains <name of sensitising substance>. May produce an allergic reaction.',
      EUH209: 'Can become highly flammable in use.',
      EUH209A: 'Can become flammable in use.',
      EUH210: 'Safety data sheet available on request.',
      EUH401: 'To avoid risks to human health and the environment, comply with the instructions for use.'
    };

    const precautionaryStatements = {
      P101: 'If medical advice is needed, have product container or label at hand.',
      P102: 'Keep out of reach of children.',
      P103: 'Read carefully and follow all instructions.',
      P201: '(Deleted) Obtain special instructions before use.',
      P202: '(Deleted) Do not handle until all safety precautions have been read and understood.',
      P203: 'Obtain, read and follow all safety instructions before use.',
      P210: 'Keep away from heat, hot surface, sparks, open flames and other ignition sources. No smoking.',
      P211: 'Do not spray on an open flame or other ignition source.',
      P212: 'Avoid heating under confinement or reduction of the desensitized agent.',
      P220: 'Keep away from clothing and other combustible materials.',
      P221: '(Deleted) Take any precaution to avoid mixing with combustibles/...',
      P222: 'Do not allow contact with air.',
      P223: 'Do not allow contact with water.',
      P230: 'Keep wetted with ...',
      P231: 'Handle and store contents under inert gas/...',
      P232: 'Protect from moisture.',
      P233: 'Keep container tightly closed.',
      P234: 'Keep only in original container.',
      P235: 'Keep cool.',
      P236: 'Keep only in original packaging; Division .. in the transport configuration.',
      P240: 'Ground/bond container and receiving equipment.',
      P241: 'Use explosion-proof [electrical/ventilating/lighting/.../] equipment.',
      P242: 'Use only non-sparking tools.',
      P243: 'Take precautionary measures against static discharge.',
      P244: 'Keep valves and fittings free from oil and grease.',
      P250: 'Do not subject to grinding/shock/friction/...',
      P251: 'Do not pierce or burn, even after use.',
      P260: 'Do not breathe dust/fume/gas/mist/vapors/spray.',
      P261: 'Avoid breathing dust/fume/gas/mist/vapors/spray.',
      P262: 'Do not get in eyes, on skin, or on clothing.',
      P263: 'Avoid contact during pregnancy/while nursing.',
      P264: 'Wash hands [and ...] thoroughly after handling.',
      P265: 'Do not touch eyes.',
      P270: 'Do not eat, drink or smoke when using this product.',
      P271: 'Use only outdoors or in a well-ventilated area.',
      P272: 'Contaminated work clothing should not be allowed out of the workplace.',
      P273: 'Avoid release to the environment.',
      P280: 'Wear protective gloves/protective clothing/eye protection/face protection/hearing protection/...',
      P281: '(Deleted)Use personal protective equipment as required.',
      P282: 'Wear cold insulating gloves and either face shield or eye protection.',
      P283: 'Wear fire resistant or flame retardant clothing.',
      P284: '[In case of inadequate ventilation] Wear respiratory protection.',
      P285: '(Deleted) In case of inadequate ventilation wear respiratory protection.',
      'P231+P232': 'Handle and store contents under inert gas/... Protect from moisture.',
      'P264+P265': 'Wash hands [and ...] thoroughly after handling. Do not touch eyes.',
      'P235+P410': '(Deleted) Keep cool. Protect from sunlight.',
      P301: 'IF SWALLOWED:',
      P302: 'IF ON SKIN:',
      P303: 'IF ON SKIN (or hair):',
      P304: 'IF INHALED:',
      P305: 'IF IN EYES:',
      P306: 'IF ON CLOTHING:',
      P307: '(Deleted) IF exposed:',
      P308: 'IF exposed or concerned:',
      P309: '(Deleted) IF exposed or if you feel unwell',
      P310: '(Deleted) Immediately call a POISON CENTER or doctor/physician.',
      P311: '(Deleted) Call a POISON CENTER or doctor/...',
      P312: '(Deleted) Call a POISON CENTER or doctor/... if you feel unwell.',
      P313: '(Deleted) Get medical advice/attention.',
      P314: '(Deleted) Get medical advice/attention if you feel unwell.',
      P315: '(Deleted) Get immediate medical advice/attention.',
      P316: 'Get emergency medical help immediately.',
      P317: 'Get emergency medical help.',
      P318: 'if exposed or concerned, get medical advice.',
      P319: 'Get medical help if you feel unwell.',
      P320: 'Specific treatment is urgent (see ... on this label).',
      P321: 'Specific treatment (see ... on this label).',
      P322: '(Deleted) Specific measures (see ...on this label).',
      P330: 'Rinse mouth.',
      P331: 'Do NOT induce vomiting.',
      P332: 'IF SKIN irritation occurs:',
      P333: 'If skin irritation or rash occurs:',
      P334: 'Immerse in cool water [or wrap in wet bandages].',
      P335: 'Brush off loose particles from skin.',
      P336: 'Thaw frosted parts with lukewarm water. Do not rub affected area.',
      P337: 'If eye irritation persists:',
      P338: 'Remove contact lenses, if present and easy to do. Continue rinsing.',
      P340: 'Remove person to fresh air and keep comfortable for breathing.',
      P341: '(Deleted) If breathing is difficult, remove victim to fresh air and keep at rest in a position comfortable for breathing.',
      P342: 'If experiencing respiratory symptoms:',
      P350: '(Deleted) Gently wash with plenty of soap and water.',
      P351: 'Rinse cautiously with water for several minutes.',
      P352: 'Wash with plenty of water/...',
      P353: 'Rinse skin with water [or shower].',
      P354: 'Immediately rinse with water for several minutes.',
      P360: 'Rinse immediately contaminated clothing and skin with plenty of water before removing clothes.',
      P361: 'Take off immediately all contaminated clothing.',
      P362: 'Take off contaminated clothing.',
      P363: 'Wash contaminated clothing before reuse.',
      P364: 'And wash it before reuse.',
      P370: 'In case of fire:',
      P371: 'In case of major fire and large quantities:',
      P372: 'Explosion risk.',
      P373: 'DO NOT fight fire when fire reaches explosives.',
      P374: '(Deleted) Fight fire with normal precautions from a reasonable distance.',
      P375: 'Fight fire remotely due to the risk of explosion.',
      P376: 'Stop leak if safe to do so.',
      P377: 'Leaking gas fire: Do not extinguish, unless leak can be stopped safely.',
      P378: 'Use ... to extinguish.',
      P380: 'Evacuate area.',
      P381: 'In case of leakage, eliminate all ignition sources.',
      P390: 'Absorb spillage to prevent material damage.',
      P391: 'Collect spillage.',
      'P301+P310': '(Deleted) IF SWALLOWED: Immediately call a POISON CENTER/doctor/...',
      'P301+P312': '(Deleted) IF SWALLOWED: call a POISON CENTER/doctor/... IF you feel unwell.',
      'P301+P316': 'IF SWALLOWED: Get emergency medical help immediately.',
      'P301+P317': 'IF SWALLOWED: Get medical help.',
      'P301+P330+P331': 'IF SWALLOWED: Rinse mouth. Do NOT induce vomiting.',
      'P302+P317': 'IF ON SKIN: Get medical help.',
      'P302+P334': 'IF ON SKIN: Immerse in cool water or wrap in wet bandages.',
      'P302+P335+P334': 'Brush off loose particles from skin. Immerse in cool water [or wrap in wet bandages].',
      'P302+P350': '(Deleted) IF ON SKIN: Gently wash with plenty of soap and water.',
      'P302+P352': 'IF ON SKIN: wash with plenty of water/...',
      'P302+P361+P354': 'IF ON SKIN: Take off Immediately all contaminated clothing. Immediately rinse with water for several minutes.',
      'P303+P361+P353': 'IF ON SKIN (or hair): Take off Immediately all contaminated clothing. Rinse SKIN with water [or shower].',
      'P304+P312': '(Deleted) IF INHALED: Call a POISON CENTER/doctor/... if you feel unwell.',
      'P304+P317': 'IF INHALED: Get medical help.',
      'P304+P340': 'IF INHALED: Remove person to fresh air and keep comfortable for breathing.',
      'P304+P341': '(Deleted) IF INHALED: If breathing is difficult, remove victim to fresh air and keep at rest in a position comfortable for breathing.',
      'P305+P351+P338': 'IF IN EYES: Rinse cautiously with water for several minutes. Remove contact lenses if present and easy to do - continue rinsing.',
      'P305+P354+P338': 'IF IN EYES: Immediately rinse with water for several minutes. Remove contact lenses if present and easy to do. Continue rinsing.',
      'P306+P360': 'IF ON CLOTHING: Rinse Immediately contaminated CLOTHING and SKIN with plenty of water before removing clothes.',
      'P308+P316': 'IF exposed or concerned: Get emergency medical help immediately.',
      'P307+P311': '(Deleted) IF exposed: call a POISON CENTER or doctor/physician.',
      'P308+P311': '(Deleted) IF exposed or concerned: Call a POISON CENTER/doctor/...',
      'P308+P313': '(Deleted) IF exposed or concerned: Get medical advice/attention.',
      'P309+P311': '(Deleted) IF exposed or if you feel unwell: call a POISON CENTER or doctor/physician.',
      'P332+P313': '(Deleted) IF SKIN irritation occurs: Get medical advice/attention.',
      'P332+P317': 'If skin irritation occurs: Get medical help.',
      'P333+P317': 'If skin irritation or rash occurs: Get medical help.',
      'P336+P317': 'Immediately thaw frosted parts with lukewarm water. Do not rub affected area. Get medical help.',
      'P337+P317': 'If eye irritation persists: Get medical help.',
      'P342+P316': 'If experiencing respiratory symptoms: Get emergence medical help immediately.',
      'P333+P313': '(Deleted) IF SKIN irritation or rash occurs: Get medical advice/attention.',
      'P335+P334': '(Deleted) Brush off loose particles from skin. Immerse in cool water/wrap in wet bandages.',
      'P337+P313': '(Deleted) IF eye irritation persists: Get medical advice/attention.',
      'P342+P311': '(Deleted) IF experiencing respiratory symptoms: Call a POISON CENTER/doctor/...',
      'P361+P364': 'Take off immediately all contaminated clothing and wash it before reuse.',
      'P362+P364': 'Take off contaminated clothing and wash it before reuse.',
      'P370+P376': 'in case of fire: Stop leak if safe to do so.',
      'P370+P378': 'In case of fire: Use ... to extinguish.',
      'P370+P380': '(Deleted) In case of fire: Evacuate area.',
      'P370+P380+P375': 'In case of fire: Evacuate area. Fight fire remotely due to the risk of explosion.',
      'P371+P380+P375': 'In case of major fire and large quantities: Evacuate area. Fight fire remotely due to the risk of explosion.',
      'P370+P372+P380+P373': 'In case of fire: Explosion risk. Evacuate area. DO NOT fight fire when fire reaches explosives.',
      'P370+P380+P375[+P378]': 'In case of fire: Evacuate area. Fight fire remotely due to the risk of explosion. [Use...to extinguish].]',
      P401: 'Store in accordance with ...',
      P402: 'Store in a dry place.',
      P403: 'Store in a well-ventilated place.',
      P404: 'Store in a closed container.',
      P405: 'Store locked up.',
      P406: 'Store in corrosive resistant/... container with a resistant inner liner.',
      P407: 'Maintain air gap between stacks or pallets.',
      P410: 'Protect from sunlight.',
      P411: 'Store at temperatures not exceeding ... °C/...°F.',
      P412: 'Do not expose to temperatures exceeding 50 °C/ 122 °F.',
      P413: 'Store bulk masses greater than ... kg/...lbs at temperatures not exceeding ... °C/...°F.',
      P420: 'Store separately.',
      P422: '(Deleted) Store contents under ...',
      'P402+P404': 'Store in a dry place. Store in a closed container.',
      'P403+P233': 'Store in a well-ventilated place. Keep container tightly closed.',
      'P403+P235': 'Store in a well-ventilated place. Keep cool.',
      'P410+P403': 'Protect from sunlight. Store in a well-ventilated place.',
      'P410+P412': 'Protect from sunlight. Do not expose to temperatures exceeding 50 °C/122°F.',
      'P411+P235': '(Deleted) Store at temperatures not exceeding ... °C/...°F. Keep cool.',
      P501: 'Dispose of contents/container to ...',
      P502: 'Refer to manufacturer or supplier for information on recovery or recycling',
      P503: 'Refer to manufacturer/supplier... for information on disposal/recovery/recycling.'
    };

    /**
     * Extracts GHS information (H/P-Statements and pictograms)
     * @param data The data to extract from
     * @return The extracted data
     */
    function getGHS(data) {
      const references = getReferences(data);
      let pictograms;
      try {
        pictograms = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="Pictogram(s)")]').map(entry => ({
          reference: references[entry.ReferenceNumber],
          data: jp.query(entry, '$.Value.StringWithMarkup[*].Markup[*]').map(entry => {
            const code = entry.URL.match(/GHS\d+/)[0];
            return {
              code,
              description: ghsPictogramText[code]
            };
          })
        }));
      } catch {
        pictograms = [];
      }
      let hStatements;
      try {
        hStatements = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="GHS Hazard Statements")]').map(entry => ({
          reference: references[entry.ReferenceNumber],
          data: jp.query(entry, '$.Value.StringWithMarkup[*]').map(entry => entry.String.match(/H\d+/)[0]).map(hCode => ({
            code: hCode,
            description: hazardStatements[hCode]
          }))
        }));
      } catch {
        hStatements = [];
      }
      //ToDo(kjappelbaum): investigate in more detail why they do not have the full P statements
      //For P statements the full sentence (with conditions) is more important than just the number
      let pStatements;
      try {
        pStatements = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="Precautionary Statement Codes")]').map(entry => ({
          reference: references[entry.ReferenceNumber],
          data: jp.query(entry, '$.Value.StringWithMarkup[*]').map(entry => entry.String.match(/(?<oneP>(?<!\+)P\d\d\d(?!\+))|(?<twoP>P\d\d\d\+P\d\d\d(?!\+))|(?<threeP>P\d\d\d\+P\d\d\d\+P\d\d\d(?!\+))/gm))[0].map(pCode => ({
            code: pCode,
            description: precautionaryStatements[pCode]
          }))
        }));
      } catch {
        pStatements = [];
      }
      return {
        pictograms,
        hStatements,
        pStatements
      };
    }
    /**
     * We combine all the results and keep
     */
    function getGHSSummary(data, options = {}) {
      const {
        sourceName
      } = options;
      const result = getGHS(data);
      if (sourceName) {
        result.pictograms = result.pictograms.filter(entry => entry.reference.sourceName.match(sourceName));
        result.hStatements = result.hStatements.filter(entry => entry.reference.sourceName.match(sourceName));
        result.pStatements = result.pStatements.filter(entry => entry.reference.sourceName.match(sourceName));
      }
      return {
        pictograms: getUnique(result.pictograms),
        hStatements: getUnique(result.hStatements),
        pStatements: getUnique(result.pStatements)
      };
    }
    function getUnique(entries) {
      const uniqueMap = {};
      entries.map(entry => entry.data.forEach(line => {
        uniqueMap[line.code] = line;
      }));
      return Object.keys(uniqueMap).map(code => uniqueMap[code]).sort((a, b) => a.code < b.code ? -1 : 1);
    }

    class CompoundData {
      constructor(data) {
        this.data = data;
      }
      getReferences() {
        return getReferences(this.data);
      }
      getExperimentalData(options) {
        return getExperimentalData(this.data, options);
      }
      getIdentifiers() {
        return getIdentifiers(this.data);
      }
      get computed() {
        return getComputedData(this.data);
      }
      /**
       * Property containing a summary of GHS information
       */
      get ghs() {
        return getGHSSummary(this.data);
      }
      getGHSSummary(options) {
        return getGHSSummary(this.data, options);
      }
      /**
       * Canonical SMILES
       */
      getSMILES() {
        return getSMILES(this.data);
      }
      /**
       * Return molecular formula in Hill notation
       */
      getFormula() {
        return getFormula(this.data);
      }
      /**
       * Return the International Chemical Identifier (InChI) computed from chemical structure using the International Union of Pure and Applied Chemistry (IUPAC) standard
       */
      getInChI() {
        return getInChI(this.data);
      }
      /**
       * International Chemical Identifier hash (InChIKey) computed from chemical structure using the International Union of Pure and Applied Chemistry (IUPAC) standard.
       */
      getInChIKey() {
        return getInChIKey(this.data);
      }
      /**
       * Detailed information about GHS information
       */
      getGHS() {
        return getGHS(this.data);
      }
      getMeltingPoint(options) {
        return getMeltingPoint(this.data, options);
      }
      getBoilingPoint(options) {
        return getBoilingPoint(this.data, options);
      }
      getDensity(options) {
        return getDensity(this.data, options);
      }
      getVaporPressure(options) {
        return getVaporPressure(this.data, options);
      }
      getSolubility() {
        return getSolubility(this.data);
      }
      getFlashPoint(options) {
        return getFlashPoint(this.data, options);
      }
      toJSON() {
        const methods = Object.entries(Object.getOwnPropertyDescriptors(CompoundData.prototype)).filter(([, descriptor]) => typeof descriptor.get === 'function').map(([key]) => key);
        const result = {};
        for (const method of methods) {
          result[method] = this[method];
        }
        return result;
      }
    }

    async function compoundDataFromCID(cid, options = {}) {
      const {
        cache
      } = options;
      let compoundData;
      if (cache?.(cid)) {
        compoundData = cache(cid);
      } else {
        const dataResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug_view/data/compound/${cid}/JSON`);
        compoundData = (await dataResponse.json()).Record;
        if (cache) {
          cache(cid, compoundData);
        }
      }
      return new CompoundData(compoundData);
    }

    function checkCompoundsResult(compounds) {
      if (!Array.isArray(compounds) || compounds.length === 0) {
        throw new Error('No compound found');
      }
      if (compounds.length !== 1) {
        throw new Error('More than one compound found');
      }
      if (!('id' in compounds[0].id)) {
        throw new Error('No compound found');
      }
    }

    async function compoundFromInchiKey(inchiKey, options = {}) {
      const {
        cache
      } = options;
      let compounds;
      if (cache?.(inchiKey)) {
        compounds = cache(inchiKey);
      } else {
        const encodedInchiKey = encodeURIComponent(inchiKey);
        const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/inchikey/${encodedInchiKey}/json`);
        compounds = (await moleculeResponse.json()).PC_Compounds;
        if (cache) {
          cache(inchiKey, compounds);
        }
      }
      checkCompoundsResult(compounds);
      return new Compound(compounds[0], {
        cache
      });
    }

    async function compoundFromName(smiles, options = {}) {
      const {
        cache
      } = options;
      let compounds;
      if (cache?.(smiles)) {
        compounds = cache(smiles);
      } else {
        const encodedName = encodeURIComponent(smiles);
        const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/name/${encodedName}/json`);
        compounds = (await moleculeResponse.json()).PC_Compounds;
        if (cache) {
          cache(smiles, compounds);
        }
      }
      checkCompoundsResult(compounds);
      return new Compound(compounds[0], {
        cache
      });
    }

    async function compoundFromSmiles(smiles, options = {}) {
      const {
        cache
      } = options;
      let compounds;
      if (cache?.(smiles)) {
        compounds = cache(smiles);
      } else {
        const encodedSmiles = encodeURIComponent(smiles);
        const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/smiles/${encodedSmiles}/json`);
        compounds = (await moleculeResponse.json()).PC_Compounds;
        if (cache) {
          cache(smiles, compounds);
        }
      }
      checkCompoundsResult(compounds);
      return new Compound(compounds[0], {
        cache
      });
    }

    class Compound {
      constructor(data, options = {}) {
        this.data = data;
        this.cache = options.cache;
      }
      getCID() {
        return this.data?.id?.id?.cid;
      }
      getData() {
        const cid = this.getCID();
        const cd = compoundDataFromCID(cid, {
          cache: this.cache
        });
        return cd;
      }
      toJSON() {
        const methods = Object.entries(Object.getOwnPropertyDescriptors(Compound.prototype)).filter(([, descriptor]) => typeof descriptor.get === 'function').map(([key]) => key);
        const result = {};
        for (const method of methods) {
          result[method] = this[method];
        }
        return result;
      }
    }
    Compound.fromSmiles = compoundFromSmiles;
    Compound.fromName = compoundFromName;
    Compound.fromInchiKey = compoundFromInchiKey;

    /**
     * We check that the hazards are present in the reagents
     * @param {object} reagent
     * @param options
     */
    async function ensureHazardAndPhysical(reagent, options = {}) {
      const {
        logger
      } = options;
      if (reagent.hazards && reagent.hStatements) return;
      if (reagent.hazards) {
        if (typeof reagent.hazards === 'string') {
          reagent.hazards = reagent.hazards.split(',').map(h => h.trim());
        }
        reagent.hStatements = reagent.hazards.map(h => getHStatementsObject()[h]);
        return;
      }
      try {
        const compound = await Compound.fromSmiles(reagent.smiles);
        const data = await compound.getData();
        const summary = data.getGHSSummary({
          sourceName: 'European'
        });
        reagent.hStatements = summary.hStatements;
        reagent.hazards = reagent.hStatements.map(h => h.code);
      } catch (error) {
        logger.error(`Could not get hazards for ${reagent.smiles}: ${error.toString()}`);
        reagent.hStatements = [];
        reagent.hazards = [];
      }
    }

    var Ir=class{copyMolecule(ft){let kt=ft.getIDCodeAndCoordinates();navigator.clipboard.writeText(`${kt.idCode} ${kt.coordinates}`);}pasteMolecule(){return null;}};var Tu="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",Ee=new Uint8Array(256);for(let d=0;d<Tu.length;d++)Ee[Tu.charCodeAt(d)]=d;function hu(d){let ft=d.length*.75,kt=d.length,mt,qt=0,d1,It,D1,at;d.at(-1)==="="&&(ft--,d.at(-2)==="="&&ft--);let L1=new ArrayBuffer(ft),s1=new Uint8Array(L1);for(mt=0;mt<kt;mt+=4)d1=Ee[d.charCodeAt(mt)],It=Ee[d.charCodeAt(mt+1)],D1=Ee[d.charCodeAt(mt+2)],at=Ee[d.charCodeAt(mt+3)],s1[qt++]=d1<<2|It>>4,s1[qt++]=(It&15)<<4|D1>>2,s1[qt++]=(D1&3)<<6|at&63;return L1;}function Nr(d){return d.toString(16).padStart(2,"0");}var _A=globalThis.devicePixelRatio||1;var $n=class{constructor(ft){this.ctx=ft,this.ctx.textAlign="left",this.ctx.textBaseline="top",this.currentFontSize=12,this.currentFont="12px sans-serif",this.ctx.font=this.currentFont,this.currentColor="#000000",this.currentLineWidth=1,this.canvasCache=new Map();}clearRect(ft,kt,mt,qt){this.ctx.clearRect(ft,kt,mt,qt);}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}getSelectionBackgroundRGB(){return 12310268;}getLineWidth(){return this.currentLineWidth;}setRGB(ft){let kt=ft>>>16&255,mt=ft>>>8&255,qt=ft>>>0&255;this.currentColor=`#${Nr(kt)}${Nr(mt)}${Nr(qt)}`,this.ctx.fillStyle=this.currentColor,this.ctx.strokeStyle=this.currentColor;}setFont(ft,kt,mt){this.currentFontSize=ft,this.currentFont=`${kt?"bold":""} ${mt?"italic":""} ${ft}px sans-serif`,this.ctx.font=this.currentFont;}getFontSize(){return this.currentFontSize;}getBounds(ft){let kt=this.ctx.measureText(ft);return {x:kt.actualBoundingBoxLeft,y:kt.actualBoundingBoxAscent,width:kt.actualBoundingBoxRight,height:kt.actualBoundingBoxAscent};}drawString(ft,kt,mt){this.ctx.fillText(mt,ft,kt);}drawCenteredString(ft,kt,mt){this.ctx.textAlign="center",this.ctx.textBaseline="middle",this.ctx.fillText(mt,ft,kt),this.ctx.textAlign="left",this.ctx.textBaseline="top";}setLineWidth(ft){this.currentLineWidth=ft,this.ctx.lineWidth=ft;}fillRectangle(ft,kt,mt,qt){this.ctx.fillRect(ft,kt,mt,qt);}fillCircle(ft,kt,mt){let qt=mt/2;this.ctx.beginPath(),this.ctx.arc(ft+qt,kt+qt,qt,0,2*Math.PI),this.ctx.fill();}drawLine(ft,kt,mt,qt){this.ctx.beginPath(),this.ctx.moveTo(ft,kt),this.ctx.lineTo(mt,qt),this.ctx.stroke();}drawPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let kt=1;kt<ft.getSize();kt++)this.ctx.lineTo(ft.getX(kt),ft.getY(kt));this.ctx.stroke();}drawRectangle(ft,kt,mt,qt){this.ctx.strokeRect(ft,kt,mt,qt);}fillPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let kt=1;kt<ft.getSize();kt++)this.ctx.lineTo(ft.getX(kt),ft.getY(kt));this.ctx.fill();}drawImage(ft,kt,mt,qt,d1,It,D1,at,L1){if(arguments.length!==9)throw new Error(`drawImage call with ${arguments.length} arguments unimplemented`);let s1=this.canvasCache.get(ft);if(!s1){s1=document.createElement("canvas");let m1=ft.imageData;s1.width=m1.width,s1.height=m1.height;let Q1=s1.getContext("2d");Q1.globalAlpha=0,Q1.putImageData(m1,0,0),this.canvasCache.set(ft,s1);}this.ctx.drawImage(s1,kt,mt,qt,d1,It,D1,at,L1);}isDarkBackground(){return  false;}};var Gr=class{constructor(ft,kt){this.canvasElement=ft,this.changeListener=kt,this.drawContext=new $n(this.canvasElement.getContext("2d"));}getBackgroundRGB(){return 16777215;}getCanvasWidth(){return this.canvasElement.width;}getCanvasHeight(){return this.canvasElement.height;}getDrawContext(){return this.drawContext;}onChange(ft,kt){this.changeListener?.({what:ft,isUserEvent:kt});}getClipboardHandler(){return new Ir();}};var u2=`
/* We can customize editor styles here. */
`,M8;function cu(){if(M8)return M8;let d=new CSSStyleSheet();return d.replaceSync(u2),M8=d,d;}function S8(d,ft,kt){let mt=-1;function qt(It,D1,at=0){D1.button>0||ft.fireMouseEvent(It,D1.button+1,at,Math.round(D1.offsetX*_A),Math.round(D1.offsetY*_A),D1.shiftKey,D1.ctrlKey,D1.altKey,D1.button===2);}d.addEventListener("pointerdown",It=>{mt===-1&&(mt=It.pointerId,qt(kt.MOUSE_EVENT_PRESSED,It));});function d1(It){mt===It.pointerId&&(mt=-1,qt(kt.MOUSE_EVENT_RELEASED,It));}return document.addEventListener("pointerup",d1),d.addEventListener("click",It=>{qt(kt.MOUSE_EVENT_CLICKED,It,It.detail);}),d.addEventListener("pointerenter",It=>{qt(kt.MOUSE_EVENT_ENTERED,It);}),d.addEventListener("pointerleave",It=>{qt(kt.MOUSE_EVENT_EXITED,It);}),d.addEventListener("pointermove",It=>{mt!==-1?mt===It.pointerId&&qt(kt.MOUSE_EVENT_DRAGGED,It):qt(kt.MOUSE_EVENT_MOVED,It);}),()=>{document.removeEventListener("pointerup",d1);};}function bu(d,ft,kt,mt,qt){let d1=typeof navigator<"u"&&navigator.platform==="MacIntel",It=at=>d1&&at.metaKey||!d1&&at.ctrlKey;function D1(at,L1){let s1=o2(L1,mt);s1!==null&&kt.fireKeyEvent(at,s1,L1.altKey,L1.ctrlKey,L1.shiftKey,It(L1));}return ft.addEventListener("keydown",at=>{It(at)&&at.key==="c"||It(at)&&at.key==="v"||D1(mt.KEY_EVENT_PRESSED,at);}),ft.addEventListener("keyup",at=>{D1(mt.KEY_EVENT_RELEASED,at);}),d.addEventListener("paste",at=>{let L1=at.clipboardData.getData("text"),s1=qt.fromText(L1);s1&&s1.getAllAtoms()>0&&kt.addPastedOrDropped(s1);}),()=>{};}function o2(d,ft){switch(d.key){case "Control":return ft.KEY_CTRL;case "Alt":return ft.KEY_ALT;case "Shift":return ft.KEY_SHIFT;case "Delete":case "Backspace":return ft.KEY_DELETE;case "F1":return ft.KEY_HELP;case "Escape":return ft.KEY_ESCAPE;case "Enter":return ft.KEY_ENTER;default:return d.key.length===1?d.key.codePointAt(0):null;}}var Kr=class{constructor(ft){this.canvasElement=ft,this.drawContext=new $n(this.canvasElement.getContext("2d"));}setDimensions(ft,kt){this.canvasElement.width=ft,this.canvasElement.style.width=`${ft/_A}px`,this.canvasElement.height=kt,this.canvasElement.style.height=`${kt/_A}px`;}getDrawContext(){return this.drawContext;}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}};var wu={"chain.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB2UlEQVR4XmNgGJmAkYGJgQULZGXAhCygIGICQg4GdjTIARRDh5xAEQwNKYzXeK4xY1Oez3SNfz8TmoYUxq+6m9bfqLjDI4tmfhzLff9VR17FoWhwZXpkOm2v1C+79+cbD4gga8lneuhrfdv6/+ItSBpSGB+bTDos9IfzP9N/k3cX6+7wwByWz/TWR/E1w/+kO+8j4BpSGL9rz9wh+Efs14TF3heY/tu/vlF2hxtkSxzLY2/FVyz/om69jT0G8kM6QyNDM9MT41nbZX9y/J/ct1/4iU7aIYb/9q8uVx0UbmF64mV7i/1/0s0HSVM5dRgYrjB84/8U+bh80iH+34z/17Xc51ZhUGX8qpkK1GL6+krltyClVyz/c+69C57CwQsK1v+Sl9rdvgr8Z/7P9WvKxNs8fsD44ABqeWLkdYHhv9sL8e+s/xJuv4k4yajEAI6H/+6llxn+g6DUh75gSXhgugEdmXIUJB5150HSFA59oHKIBteGSyBhpv+q9wsVQaZDICdDKuNnneRjCY9exU3l5AUrh2gQelAQ/8Tgve2Tb25NjIgYBkmnMl5Sumi+hQmiGKqhgaGB57Pts4Bfek0sjkjmIxQhs8BpiZFBh0EPCLGlTnQxcGod8gAAp9es6fCW8G8AAAAASUVORK5CYII=)","eraser.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAC1UlEQVR4XnWU309SYRjHn8OYTTM3zTmjC8NFFBR10dJu27wx11VX/QNdeGebTsvua93mRbUmLhURzYgKRZRJUJLibLPpRDCZC3/b1AMc4Tw97+EAR2e8FzD4fr7v53nPOQDCDkT/s2KAKiwROOXvFP8JLnDDGHhgArzgg28wCQGYgiDMw6E+1pY2rsECLEIIIrAC1O2CV/AWzPAOLGCFQbDDJ/hC+CQktU5bXWJoMGlaVv0iPAy/GeCmeBf0QB/YYECKO6kkzAl6Z/+t+BmsSJotgmmN4lFYZcA4tfdQez+1D8Fnio+S1oHe5TAcFuJHVBNit4nGLYrHGOCBbin+ntodMEzxKMkMWy8LpfgSEUN4niC7TTBtqzYYMCG3Z9xHadgD/bC1hi+U4uw1TkhFsqs3adpngJdGHSAZB8VHYAZ4vdt+RSjED3KcvYWYWMJuFY0E+MjdQe5MZgYE7YhFL8soCJzDU1iKwdcE+HOjBoG/NNJbq5DJI40EPNiIPCLgO/Wzk4lS3D2kPyaTQe5hETZsLDfxRRQLkLuLDjJd7u8wJs/mRs22J7CR4re3Qq3xIpENPUUHG+biuqX2UVsVX41epTp9ZjINm0ttqTIEZECQboK0ztd/Ya951t+nRi2GFYgksxluSlO7BMTYLVblMVcmOOTEZ9N+i4YvkREmAySz2JaNE4CqtM5jvrZHcbbE5lmLU8NrJTEmc3drsT1VlmmXdsCbgTdSu7wKUi2zPqsaq7GWZCjeJBZnwxlAN9dRyecBtsvzH/4+DV+MNdsLT4/GJSU0jHde3T2KtMxYnA8X5tvT5cp2eehDLnVxrPPoLgWpx4Hd+8Lp/Zx7boYoe/g40eDtViIF6Se+xPX8qPl96DqECImrRMOY2fg3e1atwd16UX1cR75wEXr4VmGdxL52atbpX2HnxfTenfiJcelKr9CjvQp/IMWJ57AO62ndOKk7890/iSWBU4XaZc0AAAAASUVORK5CYII=)","fist.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAABj0lEQVR4Xu2S3yuDYRTHz7IZsfdCUrYLG4VCpJV6d0mS/A1zg79ArRArP4q5s4ulVousFCn/gX/g5WK7wkZMzIXMxjZ793XeMdn7vpM/wDn1PD1P53PO95znIfq3rwmYyU5tZPr7PJYoTjGaZ8xB1t+xZQqyP60iAEeWsThFGKpqbiE9CRECrnCHLnTAj17QbLV4E3mFXALbcOAGSXRiClnMgZLkZR/QYg0UEpDBDuxI4IHzT6OADdRiEK2gE12gBWmWE8Yj7weQIGOT650zaoiqAVtjDx2KeEWR8xZ4lUt+jEWkWJxJBYgkWa7NLy5WXWl5FilzhfoKwEAzzkwYI3DhTQUoVYpYR10FYCTPaDbJ2heQUwGfR0kFELmt6Utu9Jmz6dmFpmmR7v141w1XBiCBNFNacyKmCxT59kgHGBIQ4g7UkpRzESuwaCoQ+YZxqwGUGaUwrvvStqazLeRLj/bTZUTQB5rQfg1jjac9e8qAzFgZKbDMALpj1Kz3Z0WKjiGIfex++x6H98PgK4d/APiDpdwStyBoAAAAAElFTkSuQmCC)","hand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB50lEQVR4Xq2TwUsUcRzF3yy52TboHiKVLBCElb1479g5kgKPCkKs4ME6eAiiP0GELileStew6NLJq5Sihw6BG2VlECymB2PVXd10Zl7vN+OuOjNLl35fZhiY+fzee9/vb4D/tAow9fafu90BsqrvnXzNBwSTTBAZ2I3BkSbeYBMzLHNRn9/nEFHCvYZAutDHDfb5wDvanOVnoozBhkBzIUeHOQEHfO8DW0QV0xjC3VjoUh2onAC/ZPAme4giemMQo+BKoaeusCmdSa41MmYL8HxLgUKe27q/UKLzSSxcwUOMwk7VgTKX6sBz7oeA61hJHSUP8ST1raZQA7Z8hb0QkO1WE+d5lZdlx+UbTsjSR95i0bc0E7GUzfIHdziouRqFPzwU5ii4IzOP+SFiqe3i8jyP1A1bgKPyVGa5/M2S7iaDdWZ8KUwNe1Uus923FHx8ujwZPA9YGO48+CK/Ywp4un8N8SIZgG78fCTvFV1n9w+ePe6GFMzMJ7u4zuOInQDYYaKcDB3Ba63FGQWPLk8mt2MAWKv96kk0sKM2TBFfkQ4fvkx6fyHSH9NalwO0PkXPaktirt8rhRpqelblbV6IASzkmivP9NrEdE9AM/FXbCPG4v+6p+3VcZ2ql5zTZSqv+XcQ6+gKgL8a/RiIJPdySgAAAABJRU5ErkJggg==)","handPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACAUlEQVR4Xq2Tz0sUcRjGn1lytW2wPUQamSAIK3vp7kk6i1LgMUGIFTqUBw9BiPsPhNAlpUtuKhpdOnVUKqlDh6CNsh+CsJQdjC13ddOZeXq+M7uTzu7SpXmZLwvzfr7P8/5Y4D89eZh48s/bhoC04nMXH/EmwThjRAp2c/B6C7vZwhRLXFP6NY4RRVxpCiTzw/zCYR94RpsP+Z4oYbQp0JbP0GFGwB6f+8A2UcF9jOFyQ+hkCJSrwDcZ7GcfUcDFBohRcKXQFyp8lc4s3zYzZgvwfEuBwgK/65xXRccrsXAGE7gBOxECJb4IgQfcjQAX8DJxEN/HVOJTTaEGbPsKvyJAuldNXOZZnpIdl485I0tveIkF31KuzlI6zU3ucFRzNQq/uS/MUeGOzNzm6zpLHa3ryzxQN2wBjsIT4JE6f7Co09RgHRlfAnPjXoXr7PQtmdSsNqn2eDJ4HLAw3rX3QX4nVaC534sA0bYCvdi6Je9lveb+mkLw2+PPiIKZ+WwPN3ioj9MyczQMsMNYKR5ZwfOnCzkVTq7KTpYDgrKCp5XuyGw9AOvViHoSmPhryaS7nCM+IhldvlRy92nYmaBLQWtdXqX1rn5X22OLI16xihhjBjA9q3CQJxoAFjJt5Xv6bBLdKmgmvsIOYrLxv+5uZ+WOtmqJi3pNLGj+54gN9ATAH0reBFwT+FQbAAAAAElFTkSuQmCC)","invisible.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACjUlEQVR4XnWTS09TURSF178wMUExvhIfiY+0Q6MThwwEHYk6MODAqHHgzD9gHDgWYhQBAUWMqY/SogSkCGJJNAiFUiyV2qqURkFa7cPvnJZiBXMH7Tl3rbXX2ntf5bWgyH+emPI8v8reAn8nj3r0Qr3qU78GNKghDWtEfk3ot2LKKq6AphTUjGYF26ObuqU7alabOvRQj/VEz6APKS23jvI/rY/6AD2ksCH0AG9Sq9r1QJ0W/hyREFbcOqQNcqiL/3FuIpozhJeot6J+H/VHegrci60lSA7t0VVtlRNqTvPAY4bQqxYL70LdxSsvd2l164B2qVHLVN4L0U2VhL4aQl9RveDdS9gl4IcBNWpRUShNUJxUT3OG0E/UTo4u4N0a1U9SHdRu3QOQIOw8N+0Yc/A2ZwgDeHfh3ZgZpbBH+zDTAPy7Jok6qU/K0Jht2k91CL5SVD9aHh1BvYHiCxYe1nV9w9h5bVcdk4DwGn3TmYg1Y9Rb8P2FQYVQbOM+qjPaDDwMguMw3j00Mst0narUDaUgBICPI5VgAhe5rUVguZBhhMaGOEzrrU6pQpfpdoqeB8k2hvoF7UB9mhxmsyD4WYIs0c/ifEKnoVxSEsptQke5rdQ5vGdRt4SYXTEf2lWq1l1s1GojlDiAgpmTLN4KHEKegw+NKvsco8oglApd0XtoOzEzhZmCuq2QJ7RRX3lqmO84xjaxGltUj/dVcJEwhuIqwVRptsZMZwL/wG2FPIHryijVVBnUNdJlS1ZKlmJEzkApr1JDj5KsyeJaQsR+fDlW8G/KCWyl1tgpziEIxczwVcnYcT6p5LpwO7gZ5jzH7mT0hmbW0+IO/UBixXX5L4RZlmpOn8u6vT7Y3P4BKLGmkBK1qvgAAAAASUVORK5CYII=)","lasso.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADg0lEQVR4Xk2UXWxTZRjHn7XYfWhh4DKlIFLRdlucImbGr2gEwYwsAjGRXXihRlTOFrOoF0RcdWwhEVnQdgOmbgzoliEhAZJlvZwdMQg6No/U2IYOWVrK1q1d19q1Pef8fU7bsy3/nIvznv/ved/n47xEZKDivNpK8RoE6XC4J9gX7A/2hXukwxCwta1UcxhUu4PcLFHnqZnaf+/E0FhTqDpiiaqqjjSFhsbvnZja76kRdarLQRzbTSjAuozgczaHVoJYhShX1krlSmH2bSWaQz5nRsA6MJQDrHMtx0Y3y4QS7Eo0eTpG3K5rP7tdHSNNnl2JNQxtlo+NzrXAmgXEDcnWA/5ihbAtOeyKtKf3wmIvs6+2l8GS3hv51u3aliQUKwf8yVZxA2GNIvSIBCNaJmPt/jrR6KZfyL6omQdQF2tvmTTyPt2iIhDq/f1mth/1LtjCVTa9QAIN8dEl8tK8mgJBj6oF21GvEWb4+wlft08Sds+FHbPmDl2A/mOl2Ran7SQylkV0MIcdu+cI7J3utUb0aL2q1DZSlDiRvObpKdpCZymVf1dqW6/qYY1QsM8SXYULw3jcq8XLWlSghOrJpYUwXxpeDUuUZn6smNWheUx6l7uxGD8HrKXz9A2doQidJnnPoWt6VEQItm4/4aX0LSd2wLCEqMB6+otmqYt8K1LVd3+oTRA6/yXUTXe+mCHsjE048QbWw8R6EHoNAMlGbA+d/PSOAZXKZC9tKbr19m+DFZztc/E/B2/3jp//x5logyC/8oyBd9DBIjeODj4/X8hTcO4PZZ/a6RKpYfBKWb6CuTQ+i41elD5A5cLHga7jPjOvWDIDv6e/EE25WXo49N0mPmF21OR3wm/FTahMf3/dNtLoe4GPa1LeC9y4xGUxacNnvHnykfTG+IWxJzOlyhnPzHH7bUJRvgINiSuu+CE8i2Jw8NwORq/jidRDktMz4COU46twnVyEx+brwxfFQDc+weu4P0drQFHq/Z+uP73UZq7vh8FAV6oeL2OTZl4OFMCUaZw4NXDjyNTBWENylaLDwXFpH+5Tm5mku4sN1XZQFwrkR+OvyjvxpvT5kTuEHcmpDh5i/qIQ/1l5LQd4RifUOS1A1XTnRq76r5dhzRlTPBxx7vlSDstmiA0r8NGXUcLZm6jJARkGZvnJAtqtod4JmhJbT/9tiZ4bS9QsX83dGv8DIrFVpnd+f4UAAAAASUVORK5CYII=)","lassoPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADiklEQVR4XlVUbWxTVRh+12K3oYWBy9SCSAHbbXGRaGb8SoygmJFFICayH/5AIyrdYhb1BxFXHWswQRa03WVM3RzYjaEhYcZl/Tk6QhB0bN62xjV0yNJStm7tutbetvfex3N7e3XmyUnuOXme9+O8zz1EZKDyIhwVeBE28VisLzIQGYwMxPrEY7Bhh6NCYxgUuou8DLwuUD936G736GRrtC5uSSioi7dGR6fuds8dCtTzOoXlIhbbSyjBhrwt6G6LrgExlKJKfkiskksLuzVoiwbdeRs2gIlUgXWp/eTEdomwGnvSrYGuca/n2g9eT9d4a2BPej0TbZdOTiy1w1oQ8JsyHYdD5TJhZ2bME+/M7YfFWelc56yEJbc//oXXszNDKJcPhzId/CbCetnWxxOMaJ9NdoYaeaOXLpHg8HOCw0lOWrgPjcnO9lkjy9PLyzZCU2jQzOgnpgV7rNaut5GNRsnPEXzcstICQY9awX5i2ggzQoOEzzpnCXuXYq5Fc5cuTH8z5IoCnkRVooM55tq7RGDc+X5rXI+Oq3JDCyVILhTj54bPEYbPKV+CoyAhuaHjqh7WOEUGLIm1uDCGbdOFeEoxK+Hjilvz8Ng6WBK08E31og5tk+IBNg3GFBw+zlfMoHwJjrMUpzMk7Tt6TY/qOMHeGyI8l7vpxi4YtOBq0xvJR4vUQ8FV2bo7XzekCdxfhMZ57tk8YXdyxo1XsBEmhvv93ZoAJBnxcvT0B7cNqJFn++mJspuv/zJSzcb2VOr3kVv9Uz/+6U47BI/vZ+FzlkEHi9QyMfL0cilzwfnf5IPKpFeLzSOXK4s3qDb8YXLiovg2aoT3wj2ngmZ2YskP/Zr7mDepXnow+uVWVmHBatIbsddSJtTkvrpuH28JPsPKNclvhm8Ms2sxaeYz+k8/nNucujD5WL5CPhtYOOW8RSgr3kBz+rIndRRPohwsuJrBOO16NPuA6A4MBQlV+DTWKJVhy3JT7CIf7sX7eAn3qmpNUJZ969vrj7PGNRjwTiTck23C89iqkVcKSmDKt8x8N3Tj+NyRZHNmrazDkSnxIO5RhpmhO/9G0jIoByXSI6kXpN14Vfzo+G3CrsxcFzOx4iJif1YRKwXMozNssZi189xmdutXfoJVJWaZOVJs5v/18H+/rcK7nyQI3/tRrwryTLDIVkGgvRrKm6AhvePMH5bE+cl0/cpT9dX4B08ibwJFFp9KAAAAAElFTkSuQmCC)","pointingHand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAADe0lEQVR4Xq2V+yvrYRzHnWwuaaUst9mY21iyHcVySghxJJLLDy7L2Y9S4geHThx+5ISc0vCDf0AS6fj1rPxkRJ2dJjvtjNolt1y+NMY+5/N5yonZbIunvm179nyf1/N+fy5PSMgLo6OjwxAWFgaxsbHQ2NhonJiYELy0Pqj/CgoKPqSmpjoGBgZgfHwciouLna2tre+D2uSlxXK5XJ2UlMTt7e2B2WyGjIwMiImJ+fhmADy9OjMzk7PZbHB0dAT4HXp6en6+GSA5OVktk8k4u90Ox8fHTEF3d/fvNwNIpVKm4AFACgYHB//Mzc0pUZ3o1SBPgEKhgKysLFd5ebkdlekTExMLUKUU56VVVVVhQQM9AdnZ2dDe3g5bW1tQU1MDKSkpNrTtr0qlsvT3939/NSA9PR2mp6fh7u4OhoaGWG2gXdDZ2Qnx8fFWBH3CR4P1IvMFe4feqoqKir7k5OQoSQEF2Wq1siwiwMzMDLjdbhgeHoa8vDw4OztjiiIiIqCyshLq6uoA39mRSCSKZ5DQ0FAxnuQXeutG+Yb8/Hwt1gIDHB4eMsDs7Czc399TsAFPCxzHwe7uLojFYlhdXQWDwQBKpZLDfdTeVMhra2vBaDRCV1cX2zAtLQ0IcHp6yuYWFxeZRfPz89DX18cAVIgYbNje3oaTkxPAJODwsN4BLS0tTPbGxgagTBCJRECFdnt7C+fn53B9fc0UXF1dweXlJbhcLnA4HAxmsVhYvVRUVHA8Hu85IDIyUl5SUgImk4mduKmp6T+ANiXvHz+Ag+ZJEYHpkxTgHt4VCIVCCTY009TUFNzc3MDk5CTlPRwcHLCN/A2CE6C0tNQ7IDw8nI8Z8LWhoeGWFGxubkJvby+THSyAz+d7jUEIpp46ISGB0+l0zHeKh9PpDAhACimd0QUuKirKO4BSC3uOTqPRwMXFBQvig/+BWER9C+8RLjo62jcAPayhvqPX6xmAgkf++hu0huKF2cdhivsGjIyMCAsLC81jY2PMHgIEMgiwvr4OcXFxHNaCbwDZ1NbW9q2srAz29/fZ6QOB0LrR0VHAdPd/ZwgEAhF6uaPVap/EwbMOHpTRPGUbKgfsBv5vPawJXn19/WfsR04qPMoozzg8hlEirK2tMUBzc3Pg93Zubq6uurqa9aCVlZUnz/LyMvu9tLQECwsLgPVD1f/Ds8n9AyubuM4FiwqZAAAAAElFTkSuQmCC)","rect.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACc0lEQVR4Xo2UT0hUURTGf7uipMy8GbUwigx0WbiwpCgQKmgl1CYj3Plahtg/eguFUIoamNxUCEFCGlIgEhSORmQRNeVMSkhpWcSU1KSOQ9i8vvfmpiMZOB8Xzjv3fPPd8+dertPHQICX/OAVnTRTz1GOaNXL7pQvSdTG9KHwCVJMyzlMFw1UsJFC8lmtVSi7Qr4u7SUVk1KsmCkyZtJETchUmTKh3NSYk6ZBq0a276nSXtRMGc+kfMIkiRY3shQkWiZ9wjjxMN5SEAuPQ5qegBALe07a+eiMCj9lxZ0ZJ+N483A9oQf6qbWED6rITW4IcQGaGJuT/Y0f7rm1cJliS+jnLjNkAviEDXTwVbZ/2gwjobLESKgY6iiwhI5gyw/N/T2zKtljF0CN6p39SAZbCVyhTpwVLKOUM8wGmn4lEy35yGH+Jp1TKF9nrRp3mzbOMcp9m0MhtLPrvwRDr/6/mwda6cZ4ON24Hd5z1RISfGNIAdmDnWabtHtlzxLjtY1pgucM2o8rPOUhj+UZY0qDWGEJad5y0ca8mO90LLyKA+yjkirN6QWeaO0kwjvucZw8m0MwGinLzmelUi1nP3vYQrUCH3GNY5Qp+UpbpelcQjNnWcchVWRQhc2jiPWsYTl7aZRetvTBtP5VaJW7la0c5g4npFbCbk5xSzlNKPGRUKk6vYCwScIH2ax2lagH1VJ6wxdbNVUumKUFhGFNaMxiyEk4v3Jn1VmE4LluBK804W/5zdDA/+OxCtk58dzeNjcS7vYJ/v1bzGMJnm7rPDLmkzlvdph2E5/zRs13M2BS/p3OvhoLkVSBi7jE5zn/VPCy+K/GH86K+gV5WsL6AAAAAElFTkSuQmCC)","rectPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACi0lEQVR4Xo2UT0hUQRzHP6c2KmrTnYoKjCAP+45FB0uKAqECT0FdMsKbr2OIUdE76EUpamHzUmoEGWmIgUnQn9UOWURp7ksJKSuLeCW1qesStq/fvB11lzr4vgzM/Ob3md+b3/xmuEYfA4Fe8ZMhOmmklmMclVYr/U6xpRg0Pn2I+yRpZsQ4Shd1lLGJCGHWSItIv0xsXTKXEp+0+AqZJqum1KCKqQpliXaqKnVK1Umrkr62VMjcoJpWvkprYAqvyUksRV7TlAYmcOP4S1EyPgEZegMgGfftjP3JHhf9kp5rz9pZ21+U44t6oZ9qA3yUjNygReSKoIEPC2H/oN19pxouUWKAfrqZJRtIAxvp4Jv09d9mGYtZ3lisBGooMkBHMKVd5QsRlSbfcxMl99tFUCX5zg1SwZSHI6qJ4q+wNHWGuSCmzqTXFEYMan7TeYlyBSi2ItymjXOMc9/sIQLt7C4AMiE36ka7K/HDlUXR1uh0qIcH9JCpd+OZ+u3wnisG8PjOCEOydr6S1hxJXhufBnjBsBlc5hkPeRS6bt2zbuUiWK1WJpThLReMz8vFk07GV3OQ/ZRTIXWqN11t7SLBO+5yglVmD0FppA0dZqWkcScH2MtmAVqsJ1zlOJZUbbnJ0kw+0MhZ1lEpGRnGCS2zVGgDa1nOPup5ahYNqnU+QrOYm9nGEe5wUqKVsofT3JRdTspZjMWictIFwBYJfIitcsClFHNYIr3hq7iapAW1VACMSoUmjUZsz/6dX6t2rvgKAN9xEvhRT0/oZaXg/7EYIFcnvvO4zUnEezSg79//LAbw5bYuKqs+q/Nqh2pX7oJ1UP1QAyqt73Tu1ShUShK8not8WbBPBy+LfjX+AjC0Avv8MQHTAAAAAElFTkSuQmCC)","zoom.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACiUlEQVR4XpVTTUgUYRh+nJWCxUsNevBiYGm7UjAnFyoI9ORKSxt4a71Ei9I10EPg0VO7hOE1ikQSofXiYv6QRX9bkTAjbbM/B2mz8id3Z2f/ZvbtndR0NYn9XuYbPuZ55n3f730efMQCZjCLOX4/xyJe4BXe4C3e4QNi0JDCBu8JRPkURxIMm8BjjOMJJvEUU8jadUn36D79Zs5X8JhSyb6AX/iBzxxfLMIMwx/gIcYwXRMTNa86Mhrp23QXOwx3sW9zNKKOaF5NzAlrnCWBFXAx43jE8PeC0b4U9KstJmh/tJh+dSlotJeEDYZ/Bea5GAtudoZDkm5jcCP1poejd+XhaG+6kc82kvRwiDpLwha+A1RX8K4OaDfCU038UTQGk/GA2qO4lDbFpfbEA4NJ0QA1UThktOeFDBPOLd+/qF9PtOZBDi30LNU91FBfgz+rvmaoIdUVmnZo4CxLQU3cYsLJ7G1HebvqC7nIhHJ6G7y7mmsNKTQtGjbyq5p3Dnzv2au31i24QK35cEhpriQAzbUp92AS1GKqI7odL5F1jM9bzV7KzoyVLvfbDhKAoYZ4wGp/NFKU8Br6scS9VrM7Fw8a56/8A271ovb0pkF9m2UPy2AWGd/k4sYdpal/7+fH4cTxvaPsGo6C3EXysWYWQafIR2cqSnFyV237CM6ADOowyM8Si+xkkCsztB2RIYZCFT2QB1mYVdwSSSza/N85nD1qDl27cyA76EQ1k6bqtESiRagrXFsdyBxQq+ySnfIhtbJ6gHX2bBprQvmAHwKH/GDBmRCDyibP4Od/HLcNZ0KcCSnkmLQsKEd4mmvfgTMhybGCb0zI4RPb1bSbUtnDUvHz4yGJ7BUWx2+5Iu7Rc33AhQAAAABJRU5ErkJggg==)"};var H8=Object.create(null),gu=3/4,zr=class{constructor(ft){this.HOTSPOT_32=ft.HOTSPOT_32,this.IMAGE_NAME_32=ft.IMAGE_NAME_32,this.cPointerCursor=ft.cPointerCursor,this.cTextCursor=ft.cTextCursor;}getCursor(ft){if(H8[ft])return H8[ft];if(this.IMAGE_NAME_32[ft])return this.buildCursor(ft);switch(ft){case this.cPointerCursor:return "default";case this.cTextCursor:return "text";default:throw new Error(`Unknown cursor: ${ft}`);}}buildCursor(ft){let kt=this.IMAGE_NAME_32[ft],qt=`${wu[kt]} ${this.HOTSPOT_32[ft*2]*gu} ${this.HOTSPOT_32[ft*2+1]*gu}, default`;return H8[ft]=qt,qt;}};var Jr=class{constructor(ft,kt){this.title=ft,this.rootElement=kt,this.elements=[],this.dialogElement=null;}setLayout(ft,kt){this.hLayout=Bu(ft),this.vLayout=`${Bu(kt)} auto`;}add(ft,kt,mt,qt,d1){this.elements.push({component:ft,x:kt,y:mt,x2:qt,y2:d1});}createTextField(ft,kt){return new O8(ft,kt);}createLabel(ft){return new E8(ft);}createComboBox(){return new x8();}createCheckBox(ft){return new q8(ft);}setEventConsumer(ft){this.consumer=ft;}showMessage(ft){window.alert(ft);}showDialog(ft){let kt=document.createElement("dialog"),mt=this.rootElement.getBoundingClientRect();Object.assign(kt.style,{position:"absolute",marginBlock:0,left:`${mt.left}px`,right:`${document.body.parentElement.clientWidth-mt.right}px`,top:`${this.rootElement.offsetTop+30}px`}),this.dialogElement=kt,this.rootElement.getRootNode().append(kt);let qt=document.createElement("form");qt.style.display="grid",qt.style.gridTemplateColumns=this.hLayout,qt.style.gridTemplateRows=this.vLayout,qt.addEventListener("submit",at=>{at.preventDefault(),this.consumer.fireOk(),ft();}),kt.append(qt);for(let{component:at,x:L1,y:s1,x2:m1,y2:Q1}of this.elements){let J1=document.createElement("div");m1===void 0?(J1.style.gridColumn=`${L1+1} / ${L1+2}`,J1.style.gridRow=`${s1+1} / ${s1+2}`):(J1.style.gridColumn=`${L1+1} / ${m1+2}`,J1.style.gridRow=`${s1+1} / ${Q1+2}`),J1.append(at.getElement()),qt.append(J1);}let d1=document.createElement("div");Object.assign(d1.style,{display:"flex",flexDirection:"row-reverse",gap:"15px",gridColumn:"1 / -1",gridRow:"-1"}),qt.append(d1);let It=document.createElement("button");It.type="submit",It.textContent="OK",d1.append(It);let D1=document.createElement("button");D1.type="button",D1.textContent="Cancel",D1.addEventListener("click",()=>{this.consumer.fireCancel(),ft();}),d1.append(D1),kt.showModal(),kt.addEventListener("cancel",()=>{this.consumer.fireCancel(),ft();});}disposeDialog(){this.dialogElement!==null&&(this.dialogElement.remove(),this.dialogElement=null);}},an=class{setEventHandler(ft){this.eventHandler=ft;}fireEvent(ft,kt){this.eventHandler(ft,kt);}},E8=class extends an{constructor(ft){super(),this.element=document.createElement("label"),this.setText(ft);}setText(ft){this.element.textContent=ft;}getElement(){return this.element;}},O8=class extends an{constructor(){super(),this.element=document.createElement("input"),this.element.type="text";}setText(ft){this.element.value=ft;}getText(){return this.element.value;}getElement(){return this.element;}},x8=class extends an{constructor(){super(),this.element=document.createElement("select"),this.element.addEventListener("change",()=>{this.fireEvent(2,this.element.selectedIndex);});}setEnabled(ft){this.element.disabled=!ft;}addItem(ft){let kt=document.createElement("option");kt.textContent=ft,this.element.append(kt);}getSelectedIndex(){return this.element.selectedIndex;}setSelectedIndex(ft){this.element.selectedIndex=ft;}setSelectedItem(ft){let kt=this.element.options;for(let mt=0;mt<kt.length;mt++)kt[mt].textContent===ft&&(this.element.selectedIndex=mt);}getSelectedItem(){return this.element.options[this.element.selectedIndex].textContent;}removeAllItems(){this.element.innerHTML="";}setEditable(){}getElement(){return this.element;}},q8=class extends an{constructor(ft){super();let kt=document.createElement("label"),mt=document.createElement("input");mt.type="checkbox",mt.addEventListener("change",()=>{this.fireEvent(3,mt.checked?1:0);}),kt.append(mt),kt.append(ft),this.element=kt,this.checkBox=mt;}setEnabled(ft){this.checkBox.disabled=!ft;}isSelected(){return this.checkBox.checked;}setSelected(ft){this.checkBox.checked=ft;}getElement(){return this.element;}};function Bu(d){return d.map(ft=>ft>0?`${ft}px`:"auto").join(" ");}var Oe=class{constructor(ft){this.imageData=ft,this.dataView=new DataView(ft.data.buffer);}getWidth(){return this.imageData.width;}getHeight(){return this.imageData.height;}getRGB(ft,kt){let mt=this.dataView.getInt32((kt*this.imageData.width+ft)*4,false);return (mt&255)<<24|mt>>>8;}setRGB(ft,kt,mt){let qt=mt>>>24&255,d1=mt<<8|qt;this.dataView.setInt32((kt*this.imageData.width+ft)*4,d1,false);}toDataURL(){let ft=document.createElement("canvas"),kt=ft.getContext("2d");return ft.width=this.imageData.width,ft.height=this.imageData.height,kt.putImageData(this.imageData,0,0),ft.toDataURL("image/png");}};var Zr=class{constructor(ft,kt,mt){this.canvasElement=ft,this.dialogRoot=kt,this.JavaEditorArea=mt;}register(ft){this.javaUiHelper=ft,this.cursorManager=new zr(this.JavaEditorArea,ft);}grabFocus(){this.canvasElement.focus({preventScroll:true});}setCursor(ft){this.canvasElement.style.cursor=this.cursorManager.getCursor(ft);}showHelpDialog(){}createImage(ft,kt){let mt=new ImageData(ft,kt);return new Oe(mt);}createImageFromBase64(ft,kt,mt){mt=mt.replaceAll(/%\d+%/g,D1=>"A".repeat(Number(D1.slice(1,-1))));let qt=hu(mt),d1=new Uint8ClampedArray(qt),It=new ImageData(d1,ft,kt);return new Oe(It);}createDialog(ft){return new Jr(ft,this.dialogRoot);}runLater(ft){typeof requestAnimationFrame=="function"?requestAnimationFrame(ft):typeof setImmediate=="function"?setImmediate(ft):setTimeout(ft,0);}};function ku(d,ft,kt,mt,qt,d1,It,D1){let{readOnly:at=false,initialMode:L1="molecule",initialFragment:s1=false}=ft,m1=document.createElement("div");m1.dataset.openchemlibCanvasEditor="true",Object.assign(m1.style,{width:"100%",height:"100%",display:"flex",flexDirection:"row",alignItems:"start",backgroundColor:"white",touchAction:"none",userSelect:"none",webkitUserSelect:"none"});let Q1=m1.attachShadow({mode:"open"});Q1.adoptedStyleSheets=[cu()];let J1=null;at||(J1=document.createElement("canvas"),Q1.append(J1));let o0=document.createElement("div");Object.assign(o0.style,{width:"100%",height:"100%"}),Q1.append(o0);let $1=document.createElement("canvas");$1.tabIndex=0,Object.assign($1.style,{outline:"none"}),o0.append($1),d.append(m1);let H0=new d1(new Zr($1,o0,mt)),Ot=new mt(Q2(L1,mt),new Gr($1,kt),H0);if(s1)if(L1==="molecule"){let n1=new It(0,0);n1.setFragment(true),Ot.setMolecule(n1);}else {let n1=D1.create();n1.setFragment(true),Ot.setReaction(n1);}H0.setEditorArea(Ot);let HA=at?null:new qt(Ot,new Kr(J1),H0);function PA(n1){$1.style.width=`${n1.width}px`,$1.width=Math.floor(n1.width*_A),$1.style.height=`${n1.height}px`,$1.height=Math.floor(n1.height*_A),n1.width>0&&n1.height>0&&Ot.repaint();}let xe=o0.getBoundingClientRect();PA(xe);let C1=new ResizeObserver(([n1])=>{PA(n1.contentRect);});C1.observe(o0);let m0=null,Rn=null,yt=null;at||(m0=S8($1,Ot,mt),Rn=bu(o0,$1,Ot,mt,It),yt=S8(J1,HA,mt));function H1(){m1.remove(),C1.disconnect(),m0?.(),Rn?.(),yt?.();}return {editorArea:Ot,toolbar:HA,uiHelper:H0,destroy:H1};}function Q2(d,ft){switch(d){case "molecule":return 0;case "reaction":return ft.MODE_REACTION|ft.MODE_MULTIPLE_FRAGMENTS;default:throw new Error(`Invalid initial mode: ${d}`);}}function du(d,ft,kt,mt,qt){class d1{#t;#A;#r;#e;#f;#P;constructor(D1,at={}){let{editorArea:L1,toolbar:s1,uiHelper:m1,destroy:Q1}=ku(D1,at,J1=>this.#i(J1),d,ft,kt,mt,qt);this.#t=L1,this.#A=s1,this.#r=m1,this.#e=null,this.#f={[d.EDITOR_EVENT_MOLECULE_CHANGED]:"molecule",[d.EDITOR_EVENT_SELECTION_CHANGED]:"selection",[d.EDITOR_EVENT_HIGHLIGHT_ATOM_CHANGED]:"highlight-atom",[d.EDITOR_EVENT_HIGHLIGHT_BOND_CHANGED]:"highlight-bond"},this.#P=Q1;}getMode(){return this.#n(),this.#t.getMode()&d.MODE_REACTION!==0?"reaction":"molecule";}setMolecule(D1){this.#n(),this.#t.setMolecule(D1);}getMolecule(){return this.#n(),this.#t.getMolecule();}setReaction(D1){this.#n(),this.#t.setReaction(D1);}getReaction(){return this.#n(),this.#t.getReaction();}setOnChangeListener(D1){this.#n(),this.#e=D1;}removeOnChangeListener(){this.#n(),this.#e=null;}clearAll(){this.#n(),this.#t.clearAll();}destroy(){this.#n(),this.#P(),this.#t=null,this.#A=null,this.#r=null,this.#e=null,this.#P=null;}get isDestroyed(){return !this.#t;}moleculeChanged(){this.#n(),this.#t.moleculeChanged();}#n(){if(this.isDestroyed)throw new Error("CanvasEditor has been destroyed");}#i(D1){if(!this.#e)return;let{what:at,isUserEvent:L1}=D1;this.#e({type:this.#f[at],isUserEvent:L1});}}return d1;}function mu(d,ft,kt,mt){class qt extends HTMLElement{static MODE=Object.freeze(Object.create({MOLECULE:"molecule",REACTION:"reaction"}));static observedAttributes=Object.freeze(["idcode","fragment","mode","readonly"]);#t={idcode:"",fragment:false,mode:qt.MODE_MOLECULE,readonly:false};get idcode(){return this.#t.idcode;}set idcode(It){this.#t.idcode=String(It),this.setAttribute("idcode",this.#t.idcode);}get fragment(){return this.#t.fragment;}set fragment(It){this.#t.fragment=!!It,this.#t.fragment?this.setAttribute("fragment",""):this.removeAttribute("fragment");}get mode(){return this.#t.mode;}set mode(It){this.#t.mode=String(It),this.setAttribute("mode",this.#t.mode);}get readonly(){return this.#t.readonly;}set readonly(It){this.#t.readonly=!!It,this.#t.readonly?this.setAttribute("readonly",""):this.removeAttribute("readonly");}setMolecule(It){this.fragment=It.isFragment(),this.idcode=`${It.getIDCode()} ${It.getIDCoordinates()}`,this.#A.setMolecule(It);}getMolecule(){return this.#A.getMolecule();}setReaction(It){this.fragment=It.isFragment(),this.idcode=mt.encode(It,{keepAbsoluteCoordinates:true,mode:mt.INCLUDE_MAPPING|mt.INCLUDE_COORDS|mt.RETAIN_REACTANT_AND_PRODUCT_ORDER})??"",this.#A.setReaction(It);}getReaction(){return this.#A.getReaction();}clearAll(){this.#A.clearAll(),this.idcode="";}moleculeChanged(){this.#A.moleculeChanged();}#A;#r(){this.#A||(this.#A=new d(this,{readOnly:this.readonly,initialMode:this.mode}),this.#A.setOnChangeListener(this.#Q),requestIdleCallback(()=>this.#e()));}#e(){switch(this.mode){case qt.MODE.MOLECULE:return this.#P();case qt.MODE.REACTION:return this.#n();default:throw new Error(`Mode ${this.mode} is not supported`);}}#f(It){let D1=It.indexOf(" ");if(D1===-1)return ft.fromIDCode(It);let at=It.slice(0,D1),L1=It.slice(D1+1);return ft.fromIDCode(at,L1);}#P(){let It=this.#f(this.idcode);It.setFragment(this.fragment),this.#A.setMolecule(It);}#n(){let It=mt.decode(this.idcode,{ensureCoordinates:true})??kt.create();It.setFragment(this.fragment),this.#A.setReaction(It);}#i=false;#o(It){this.#i=true;try{It();}finally{this.#i=false;}}#Q=It=>{let D1=this.idcode,at=this.fragment;this.#o(()=>{if(It.type==="molecule")switch(this.mode){case qt.MODE.MOLECULE:{let s1=this.getMolecule();this.idcode=`${s1.getIDCode()} ${s1.getIDCoordinates()}`,this.fragment=s1.isFragment();break;}case qt.MODE.REACTION:{let s1=this.getReaction();this.idcode=mt.encode(s1,{keepAbsoluteCoordinates:true,mode:mt.INCLUDE_MAPPING|mt.INCLUDE_COORDS|mt.RETAIN_REACTANT_AND_PRODUCT_ORDER}),this.fragment=s1.isFragment();break;}default:throw new Error(`Unsupported mode ${this.mode}`);}});let L1=new CustomEvent("change",{detail:It,bubbles:true});if(this.dispatchEvent(L1),It.mode==="molecule"){if(this.idcode!==D1){let s1=new CustomEvent("idcode-changed",{detail:this.idcode,bubbles:true});this.dispatchEvent(s1);}if(this.fragment!==at){let s1=new CustomEvent("fragment-changed",{detail:this.fragment,bubbles:true});this.dispatchEvent(s1);}}};#D(){this.#A&&(this.#A.destroy(),this.#A=void 0);}#u(){this.#D(),this.#r();}connectedCallback(){this.#t={idcode:this.getAttribute("idcode")||"",fragment:this.hasAttribute("fragment"),mode:this.getAttribute("mode")||qt.MODE.MOLECULE,readonly:this.hasAttribute("readonly")},this.#r();}disconnectedCallback(){this.#D();}adoptedCallback(){this.connectedCallback();}attributeChangedCallback(It,D1,at){if(!this.#A||this.#i)return;(()=>{switch(It){case "idcode":return this.#t.idcode=String(at),()=>this.#e();case "fragment":return this.#t.fragment=at!==null,()=>this.#e();case "mode":return this.#t.mode=String(at),()=>this.#u();case "readonly":return this.#t.readonly=at!==null,()=>this.#u();default:throw new Error("unsupported attribute change");}})()();}}return qt;}function V8(d){let{GenericEditorArea:ft,GenericEditorToolbar:kt,GenericUIHelper:mt,Molecule:qt,Reaction:d1,ReactionEncoder:It}=d,D1=du(ft,kt,mt,qt,d1);function at(){let L1=customElements.get("openchemlib-editor");if(L1)return L1;let s1=mu(D1,qt,d1,It);customElements.define("openchemlib-editor",s1);let m1=document.createElement("style");return m1.id="openchemlib-editor-default-style",m1.innerHTML=`
    /* dynamicaly added from openchemlib registerCustomElement with low priority */
    openchemlib-editor:defined {
      display: block;
      height: 400px;
      width: 600px;
    }
    `,document.head.prepend(m1),s1;}d.CanvasEditor=D1,d.registerCustomElement=at,delete d.GenericEditorArea,delete d.GenericEditorToolbar,delete d.GenericUIHelper;}function Fu(d){d.prototype.getNextCustomAtomLabel=function(kt){let mt=kt||"1",qt=new Set();for(let It=0;It<this.getAllAtoms();It++){let D1=this.getAtomCustomLabel(It);D1&&qt.add(D1);}let d1=0;for(;qt.has(mt)&&d1++<100;)mt=l2(mt);return mt;};}function l2(d){let ft=d.match(/(\d+)/);if(ft){let mt=Number.parseInt(ft[1],10);return d.replace(/(\d+)/,(mt+1).toString());}let kt=d.match(/([a-yA-Y])([^a-zA-Z]*)$/);if(kt){let mt=kt[1],qt=String.fromCodePoint(mt.codePointAt(0)+1);return qt==="Z"||qt==="z"?"1":d.replace(/([a-yA-Y])([^a-zA-Z]*)$/,`${qt}$2`);}return "1";}function Yr(d,ft){switch(ft){case "superscript":for(let kt=0;kt<d.getAllAtoms();kt++){let mt=d.getAtomCustomLabel(kt);mt&&!mt.startsWith("]")&&d.setAtomCustomLabel(kt,`]${mt}`);}break;case "normal":for(let kt=0;kt<d.getAllAtoms();kt++){let mt=d.getAtomCustomLabel(kt);mt?.startsWith("]")&&d.setAtomCustomLabel(kt,mt.slice(1));}break;case "auto":for(let kt=0;kt<d.getAllAtoms();kt++){let mt=d.getAtomCustomLabel(kt);mt&&(d.getAtomLabel(kt)==="C"?mt.startsWith("]")&&d.setAtomCustomLabel(kt,mt.slice(1)):mt.startsWith("]")||d.setAtomCustomLabel(kt,`]${mt}`));}break;}}function vu(d){let ft=d.fromMolfile;d.fromMolfile=function(mt,qt={}){let{customLabelPosition:d1}=qt,It=ft.call(this,mt),D1=mt.includes(`\r
`)?`\r
`:`
`,at=mt.split(D1);if(at.length<4||!at[3].includes("V2000"))return It;let L1=at.slice(4+It.getAllAtoms()+It.getAllBonds());for(let s1=0;s1<L1.length;s1++){let m1=L1[s1];if(m1.startsWith("A  ")){let Q1=Number(m1.slice(3)),J1=L1[s1+1]?.trim();s1++,!Number.isNaN(Q1)&&Q1<=It.getAllAtoms()&&J1&&!It.getAtomCustomLabel(Q1-1)&&It.setAtomCustomLabel(Q1-1,J1);}if(m1.startsWith("V  ")){let Q1=m1.split(" ").filter(Boolean);if(Q1.length>=3){let J1=Number(Q1[1]),o0=Q1.slice(2).join(" ");!Number.isNaN(J1)&&J1<=It.getAllAtoms()&&!It.getAtomCustomLabel(J1-1)&&It.setAtomCustomLabel(J1-1,o0);}}if(m1.startsWith("M  ZZC")){let Q1=Number(m1.slice(7,10).trim()),J1=m1.slice(10).trim();Q1&&J1&&It.setAtomCustomLabel(Q1-1,J1);}}return Yr(It,d1),It;};}var U2=["M  STY","M  SLB","M  SAL","M  SDT","M  SDD","M  SED"];function ju(d){let ft=d.prototype.toMolfile;d.prototype.toMolfile=function(mt={}){let qt=this.getCompactCopy(),{includeCustomAtomLabelsAsALines:d1=false,includeCustomAtomLabelsAsVLines:It=false,customLabelPosition:D1,removeCustomAtomLabels:at=false}=mt;Yr(qt,D1);let L1=ft.call(qt);if(!d1&&!It&&!at)return L1;let s1=L1.includes(`\r
`)?`\r
`:`
`,m1=L1.split(s1);if(at&&(m1=m1.filter(o0=>!U2.some($1=>o0.startsWith($1)))),m1.length<4||!m1[3].includes("V2000"))return L1;let Q1=[];for(let o0=0;o0<qt.getAllAtoms();o0++){let $1=qt.getAtomCustomLabel(o0);if($1){let H0=String(o0+1).padStart(3," ");d1&&Q1.push(`A  ${H0}`,$1),It&&Q1.push(`V  ${H0} ${$1}`);}}let J1=m1.findIndex(o0=>o0.startsWith("M  END"));return J1===-1?L1:(m1.splice(J1,0,...Q1),m1.join(s1));};}function pu(d){let ft=d.prototype.toRxn,kt=d.prototype.toRxnV3;d.prototype.toRxn=function(qt={}){return Ru(this,ft,qt);},d.prototype.toRxnV3=function(qt={}){return Ru(this,kt,qt);};}function Ru(d,ft,kt){let{programName:mt="",keepIdCode:qt=false}=kt,d1=ft.call(d,mt);return qt?d1:d1.replace(/^OCL_RXN_V1.0:.*$/m,"");}function Cu(d){let{ConformerGenerator:ft,ForceFieldMMFF94:kt,Molecule:mt,Reaction:qt}=d;ft.prototype.molecules=function*(){let L1;for(;(L1=this.getNextConformerAsMolecule())!==null;)yield L1;};let d1={maxIts:4e3,gradTol:1e-4,funcTol:1e-6},It=kt.prototype._minimise;delete kt.prototype._minimise,kt.prototype.minimise=function(L1){return L1={...d1,...L1},It.call(this,L1.maxIts,L1.gradTol,L1.funcTol);},vu(mt),ju(mt),pu(qt),Fu(mt);function D1(at){if(!at)return null;if(at.includes("V2000")||at.includes("V3000"))return mt.fromMolfile(at);try{return mt.fromSmiles(at);}catch{}try{return mt.fromIDCode(at);}catch{}return null;}mt.fromText=function(L1){let s1=D1(L1);return s1&&s1.getAllAtoms()>0?s1:null;},mt.prototype.getOCL=function(){return d;};}function jn(){}function L2(d){let ft=d.document,kt={},mt={userAgent:"webkit"},qt=jn;qt.__moduleStartupDone=jn;var at="object",L1="anonymous",s1="fnStack",m1=`
`,Q1={4:1,9:1,5:1,8:1},J1="Unknown",o0="boolean",$1="number",H0="string",Ot=2147483647,HA='For input string: "',PA="null",xe="__noinit__",C1={4:1,5:1,8:1},m0={4:1,19:1,33:1,27:1},Rn=" (copy)",yt={4:1,9:1,14:1,5:1,11:1,8:1,13:1},H1=65536,n1=65535,Z0=1e4,te="fromIndex: ",I8=" > toIndex: ",Xr=", toIndex: ",N8=", length: ",_r="Index: ",Wr=", Size: ",jt={15:1,4:1,5:1},fn={4:1},H={7:1,4:1,5:1},WA=16777215,Y0=65280,Dn=.30000001192092896,EA=545460846592,lA=4294967296,OA=8589934592,xA=17179869184,qA=34359738368,VA=68719476736,IA=137438953472,un=274877906944,Tt={12:1,4:1,5:1},i0=536870912,$r=2.617993878,ar=3.665191429,N1=6.283185307179586,t1=3.141592653589793,F0=1.5707963267948966,N0=16384,qe=-8355712,Z1=4096,Ae={l:0,m:0,h:1},ne={l:0,m:0,h:2},ee={l:0,m:0,h:4},X0=1920,f0=1024,mA=234881024,re=167772160,on=100663296,Qn=201326592,$A=114688,aA=4063232,Pe=3801088,pn=3538944,Cn=3014656,ie=3145728,j0=2097152,tn=393216,iA=917504,ln=1966080,Un=0xf8000000000,Ve=0xf0000000000,Ie=0xe8000000000,Ne=0xd8000000000,Ge=0xb8000000000,tP=0xe0000000000,AP=0xc0000000000,fe=8796093022208,Mn=549755813888,Ke=1649267441664,ze=3848290697216,De=8246337208320,nP=0xc8000000000,eP=9345848836096,rP=0x98000000000,FA=29360128,NA=268435456,U0=8192,B1=2048,UA=32768,ue=-1.5707963267948966,LA=130560,oe=6291456,PP=-2.0943951023931953,c0=1.0471975511965976,Sn=.5235987755982988,l1=786432,sA=524288,yA=262144,G0=-16777216,fA=-65536,Je=-32768,iP='<DrawingObject type="',Ze="></DrawingObject>",Yt={4:1,5:1,16:1},e0={4:1,5:1},fP="unsupported atomicNo:",G8=2.8415926535897933,E1={28:1,4:1,9:1,5:1,8:1},Hn={4:1,5:1,11:1,8:1},Y={10:1,4:1,5:1},b0={4:1,9:1,5:1,8:1,30:1},r0=131072,Ye=196608,Xe={l:6,m:0,h:4},TA=1048576,hA=3221225472,DP={l:0,m:0,h:3},Ln=8388608,uP={l:0,m:0,h:120},_e={36:1,43:1,38:1},y0={4:1,9:1,5:1,11:1,8:1},K8={36:1,43:1,95:1,38:1,83:1},We={4:1,5:1,36:1,43:1,95:1,20:1,38:1,83:1,126:1},L0=17976931348623157e292,oP=-17976931348623157e292,GA={87:1,73:1,74:1,4:1,9:1,5:1,8:1},p1="??",cA={58:1,4:1,9:1,5:1,8:1},bA=2013265920,v0=32505856,E0=31744,$e=122880,En={128:1,4:1,9:1,5:1,11:1,8:1},Qe=33554432,O0=67108864,ae=-131073,tr=-33554433,Ar=-33292289,QP=-3.141592653589793,W1=4194303,x0=3072,w0={46:1,4:1,5:1},lP=.7853981633974483,z8=3.061592653589793,q0=1073741824,J8=541165879296,xt={17:1,4:1,5:1,8:1},nr=281.17,er=289.19,rr=294.21,Pr="0.0000",ir="NOSEARCH_OCL_CUSTOM_LABEL",vA="M  END",On="ATOMS",e1=-2147483648,le="$$$$",Ue="M  V30 ",Z8="M  V30 COUNTS ",Y8="M  V30 MDLV30/STEREL",X8="M  V30 MDLV30/STERAC",UP="M  V30 END CTAB",_8=34028234663852886e22,LP={l:2,m:0,h:4},_0=4194304,sP={l:4179966,m:4063288,h:7},W8={l:4194302,m:4193343,h:127},Y1='" ',yP='stroke="',fr='stroke-width="',Dr='" />',$8='class="event" ',a8='opacity="0" />',W0=16711680,ti="' position:",Ai="'. Position:",ni=1099511627776,ei=2199023255552,ri=4398046511104,TP="', position:",Pi="Assignment of aromatic double bonds failed",Le=2147483646,ii={102:1,4:1},hP="Members of ESR groups must only be stereo centers with known configuration.",cP="Over- or under-specified stereo feature or more than one racemic type bond",bP="Ambiguous configuration at stereo center because of 2 parallel bonds",fi={114:1,88:1,26:1,4:1},ur=1e5,se=1.399999976158142,xn=3.1415927410125732,wP=1.5707963705062866,Di=.029999999329447746,ui=6.2831854820251465,oi=1.9106119871139526,Qi=2.2639999389648438,gP=2.16759991645813,li=2.0662999153137207,Ui=2.3561999797821045,Li=2.2844998836517334,si=1.9106120066311025,yi=2.094395160675049,ye=.6000000238418579,or=1.7999999523162842,BP=.05000000074505806,P0={4:1,9:1,5:1,8:1,37:1},Qr=.2617993950843811,qn=2.0943951023931953,Te=1.2000000476837158,kP=1.4299999475479126,lr=1.659999966621399,dP=1.4600000381469727,mP=1.899999976158142,Ti=1.8200000524520874,hi=2.4600000381469727,ci=2.4200000762939453,Ur=2.450000047683716,FP=2.440000057220459,bi=2.4000000953674316,vP=2.319999933242798,jP=2.2899999618530273,wi=1.8600000143051147,gi=2.5299999713897705,Bi=2.490000009536743,ki=2.4700000286102295,di=2.880000114440918,mi=2.9200000762939453,Fi=2.869999885559082,RP=2.8299999237060547,vi=2.7899999618530273,ji=2.809999942779541,Ri=2.799999952316284,pi=3.049999952316284,pP=.3199999928474426,Ci=1.3300000429153442,he=.8500000238418579,Mi=1.600000023841858,Si=1.3899999856948853,CP=1.159999966621399,Hi=1.1100000143051147,Ei=1.0700000524520874,Oi=1.9600000381469727,xi=1.7100000381469727,MP=1.3600000143051147,qi=1.340000033378601,Vi=1.2200000286102295,Ii=1.2400000095367432,Lr=1.2100000381469727,Ni=2.0999999046325684,SP=1.850000023841858,HP=1.6299999952316284,Gi=1.4700000286102295,Ki=1.2799999713897705,zi=1.440000057220459,Ji=1.7599999904632568,Zi=1.7300000190734863,Yi=1.7200000286102295,sr=1.6799999475479126,Xi=1.690000057220459,_i=1.6699999570846558,EP=1.649999976158142,ce=1.7000000476837158,Wi=1.6200000047683716,$i=1.3700000047683716,ai=1.2899999618530273,t9=1.4500000476837158,A9=1.5700000524520874,OP=-0.5235987755982988,be=-1.0471975511965976,n9=-0.7853981633974483,e9=.017453292519943295,r9=.17453292519943295,we=1e-4,sn={149:1},yn=57.29577951308232,P9=.021922173386725217,ge=1e-8,DA=2.5120960256267386,Be=1e-5,yr="MMFF94s+",i9="nonbonded cutoff",f9="dielectric constant",D9="dielectric model",u9="angle bend",o9="bond stretch",Q9="electrostatic",l9="out of plane",U9="stretch bend",L9="torsion angle",s9="van der waals",y9=.07000000000000006,T9=.1200000000000001,h9=-7.840000000000001,Tr="OCL_RXN_V1.0:",c9="$RXN V3000",b9="'$MOL' tag not found",w9="'M  END' not found",g9="M  V30 BEGIN REACTANT",B9=`
Actelion Java MolfileCreator 2.0

  0  0  0  0  0  0              0 V3000
`,k9="M  V30 BEGIN PRODUCT",d9=2475109711874,m9=2475109711875,F9=2887426572290,v9=2887426572291,j9=2887627898882,R9=2887627898883,p9=2887627997186,C9=3437182386178,M9=3437182386179,S9=3437383712770,H9=3437383811074,E9=3437652148226,O9=3437652377602,x9=4674132967426,q9=4674334294018,V9=4674334392322,I9=4674602729474,N9=4674602958850,G9=5086449827842,K9=5086449827848,z9=5086651154434,J9=5086651252738,Z9=5086651252743,Y9=5086919589890,X9=5086919688194,_9=5636406968322,W9=5636407066626,$9=5636675502082,a9={l:589826,m:590112,h:4},tf={l:589827,m:590112,h:4},Af={l:589826,m:590160,h:4},nf={l:589827,m:590160,h:4},ef={l:688130,m:590160,h:4},rf={l:589826,m:590224,h:4},Pf={l:589827,m:590224,h:4},ff={l:819203,m:590224,h:4},Df={l:589826,m:590368,h:4},uf={l:688130,m:590368,h:4},of={l:819202,m:590368,h:4},Qf={l:819208,m:590368,h:4},lf={l:589826,m:590416,h:4},Uf={l:589832,m:590416,h:4},Lf={l:688130,m:590416,h:4},sf={l:819202,m:590416,h:4},yf={l:589826,m:598304,h:4},Tf={l:589827,m:598304,h:4},hf={l:688130,m:598304,h:4},cf={l:1114114,m:598304,h:4},bf={l:1212418,m:598304,h:4},wf={l:589826,m:598308,h:4},gf={l:589826,m:688416,h:4},Bf={l:589826,m:688464,h:4},kf={l:589826,m:688720,h:4},df={l:688130,m:688720,h:4},mf={l:589826,m:590112,h:288},Ff={l:589826,m:590112,h:336},vf={l:589826,m:688416,h:336},jf={l:589826,m:688464,h:336},Rf={l:688130,m:688464,h:336},pf={l:589826,m:590112,h:400},Cf={l:589826,m:688416,h:400},Mf={l:589826,m:688464,h:400},Sf={l:688130,m:688464,h:400},Hf={l:589826,m:819488,h:400},Ef={l:589826,m:819536,h:400},Of={l:589826,m:819600,h:400},xf={l:688130,m:819600,h:400},qf={l:819202,m:819600,h:400},Vf={l:589831,m:688416,h:592},If={l:589831,m:688464,h:592},Nf={l:688135,m:688464,h:592},Gf={l:589831,m:819536,h:592},Kf={l:688135,m:819536,h:592},zf={l:688135,m:819600,h:592},Jf={l:589832,m:1212704,h:592},Zf={l:589832,m:1212752,h:592},Yf={l:688136,m:1212752,h:592},Xf={l:589831,m:688464,h:656},_f={l:589826,m:590112,h:8480},Wf={l:589826,m:688416,h:8480},$f={l:589826,m:688464,h:8480},af={l:688130,m:688464,h:8480},tD={l:819202,m:819600,h:8480},AD={l:688135,m:1212752,h:8480},nD={l:589832,m:1213008,h:8480},eD={l:688136,m:1213008,h:8480},rD={l:589826,m:590112,h:8484},PD={l:589826,m:590160,h:8484},iD={l:688130,m:598304,h:8484},fD=.6262000203132629,DD=-1.3825000524520874,hr=.33169999718666077,uD=-1.4915000200271606,oD=.3540000021457672,QD=.38179999589920044,lD=-0.6019999980926514,UD=.07999999821186066,LD=-0.7379999756813049,sD="Atom-types are 64-bit numbers describing atoms and their near surrounding.",xP="Recognized atom types and their contributions are:",yD="Druglikeness predictor not properly initialized.",TD=3.009999990463257,hD=-0.17000000178813934,cD=-0.1809999942779541,bD=-0.2029999941587448,ke=2147483648,wD="ene-Metathesis",gD="Halogenation ring closure",BD=" (rootPairSets:",qP=-1e10,cr={204:1,4:1,9:1,5:1,8:1,30:1},br=-99999,wA="any",kD="is not in a ring",VP="Excluded atoms:",wr="exactly 1",gr="exactly 2",IP="less than 3",dD="less than 4",Br="at least 1",NP="at least 2",GP="at least 3",mD="at least 4",KP="1 or 2",FD="less than 2",vD="Allowed atoms:",Tn={138:1},jD="Atom Properties",kr=-8346432,RD="/html/editor/editor.html",pD="Structure Editor Help",dr="Dummy look and feel",Vn="Substance Graphite",In="Radiance Graphite",Nn="Radiance Night",CD=.6499999761581421,MD=.3499999940395355,zP="editorButtons.png",SD="esrButtons.png",hn=.800000011920929,JP=-1774190527,ZP=-1438646166,Gn=-1433403286,mr=-1431655766,Fr=-1448432982,jA=1073741866,cn=1073741930,YP="MMFF94",XP="MMFF94s",HD="/resources/forcefield/mmff94/94s/outofplane.csv",Kn="undefined",_P={172:1,4:1,5:1,8:1},ED=.08726646502812703,OD="overlapping",WP=.003921568859368563,p0=1048575,$0=17592186044416,bn=1e9,zn=-17592186044416,g0=5.56,D0=11.12,RA=13.34,wn=14.44,de=.3010299956639812,xD="BigInteger divide by zero",P1=4294967295,qD={l:0,m:0,h:524288},vr={32:1,115:1},$P={107:1,110:1},Jn={36:1,43:1,38:1,85:1},jr=16777619,VD={36:1,38:1,85:1},Rr="delete",pr=15525485,ID=5960464477539063e-23,aP=16777216,Cr={4:1,32:1,56:1,80:1},t8="Invalid UTF8 sequence",A8=.20000000298023224,ND=" distance:",GD={4:1,5:1,11:1,8:1,463:1},KD="gGP`@dfyjidNcGI[WQCP`<",Zn={49:1},Q={},h={},h1={},Kt={},w={},uA={},n={},s={},q={},Xt={},X={},gA={},Z={},rt={},r1={},A1={},et={},tt={},a={},M={},ht={},F1={},m={},W={},C={},x1={},Ut={},R1={},B0={},Ct={},p={},O1={},Q0={},l0={},K0={},Ft={},_t={},_={},$={},L,me,Mr;Q.zfb=function(){};function Hu(f,t){typeof window===at&&typeof window.$gwt===at&&(window.$gwt[f]=t);}function Eu(f,t,A,e){Q.mgb();var r=Mr;function P(){for(var i=0;i<r.length;i++)r[i]();}if(f)try{Au(P)();}catch(i){f(t,i);}else Au(P)();}Q.mgb=function(){Mr==null&&(Mr=[]);};function Ou(){Q.mgb();for(var f=Mr,t=0;t<arguments.length;t++)f.push(arguments[t]);}Q.kgb=function(){};function Yn(){}function Xn(f){return Array.isArray(f)&&f.yd===Yn?h.Uc(f):f.toString();}function V1(f,t){var A=d;if(f==="")return A;var e=f.split(".");if(!(e[0]in A)&&A.execScript&&A.execScript("var "+e[0]),t){var r=t.prototype.wd;r.e=t;}for(var P;e.length&&(P=e.shift());)A=A[P]=A[P]||!e.length&&t||{};return A;}Q.xgb=function(t){function A(){}return A.prototype=t||{},new A();},Q.wgb=function(t){return t instanceof Array?t[0]:null;};function T0(f,t,A){var e=function(){return f.apply(e,arguments);};return t.apply(e,A),e;}function i1(){}function a0(f,t){for(var A in t)t[A].configurable=true;Object.defineProperties(f,t);}function I(f,t,A){var e=me,r=e[f],P=Q.wgb(r);r&&!P?L=r:(L=Q.rgb(t),L.xd=A,!t&&(L.yd=Yn),e[f]=L);for(var i=3;i<arguments.length;++i)arguments[i].prototype=L;P&&(L.wd=P);}Q.rgb=function(t){var A=t&&t.prototype;return !A&&(A=me[t]),Q.xgb(A);},Q.qgb=function(){},d.goog=d.goog||{},d.goog.global=d.goog.global||d,me={},h.Jc=function(t,A){return j1(t)===j1(A);},h.Kc=function(t){return t.wd;},h.Lc=function(t){return w.AAb(t);};function zD(){}h.Oc=function(t,A){return pA(t)?h.ckb(t,A):gn(t)?h.Uib(t,A):Fe(t)?h.Mhb(t,A):e8(t)?t.mb(A):n8(t)?h.Jc(t,A):h1.l5(t,A);},h.Qc=function(t){return pA(t)?h.gkb():gn(t)?h.Vib():Fe(t)?h.Nhb():e8(t)?t.wd:n8(t)?h.Kc(t):h1.m5(t);},h.Sc=function(t){return pA(t)?h.hkb(t):gn(t)?h.Wib(t):Fe(t)?h.Ohb(t):e8(t)?t.ob():n8(t)?h.Lc(t):h1.n5(t);},h.Uc=function(t){var A;return h.oib(h.Qc(t))+"@"+(A=w.UAb(w.TAb(h.Sc(t))),A.toString(16));},I(1,null,{},zD),L.mb=function(t){return h.Jc(this,t);},L.nb=function(){return h.Kc(this);},L.ob=function(){return h.Lc(this);},L.pb=function(){var t;return h.oib(h.Qc(this))+"@"+(t=w.UAb(w.TAb(h.Sc(this))),t.toString(16));},L.equals=function(f){return this.mb(f);},L.hashCode=function(){return this.ob();},L.toString=function(){return this.pb();},h1.l5=function(t,A){return h1.r5(t)?h1.o5(t,A):j1(t)===j1(A);},h1.m5=function(t){return t.wd||Array.isArray(t)&&v(Q.Tbb,1)||Q.Tbb;},h1.n5=function(t){return h1.s5(t)?h1.p5(t):w.AAb(t);},h1.o5=function(t,A){return t.equals(A);},h1.p5=function(t){return t.hashCode();},h1.q5=function(){return [];},h1.r5=function(t){return !!t&&!!t.equals;},h1.s5=function(t){return !!t&&!!t.hashCode;},h1.t5=function(t){return t.toString?t.toString():"[JavaScriptObject]";},Kt.Z5=function(){Kt.Z5=i1;var t,A;A=!Kt.g6(),t=new Kt.p6(),Kt.Y5=A?new Kt.h6():t;},Kt.$5=function(t){Kt.Z5(),Kt.Y5.Nc(t);},Kt._5=function(t){Kt.Z5();var A;return A=Kt.Y5.Oc(t),Kt.a6(A);},Kt.a6=function(t){var A,e,r,P;for(A="$5",e="ZB",P=d.Math.min(t.length,5),r=P-1;r>=0;r--)if(h.ckb(t[r].d,A)||h.ckb(t[r].d,e)){t.length>=r+1&&w.UAb(t).splice(0,r+1);break;}return t;},Kt.b6=function(t){var A=/function(?:\s+([\w$]+))?\s*\(/,e=A.exec(t);return e&&e[1]||L1;},Kt.c6=function(t){return Kt.Z5(),t&&t[s1]?t[s1]:[];},Kt.d6=function(t){return Kt.Z5(),t.name||(t.name=Kt.b6(t.toString()));},Kt.e6=function(t){return Kt.Z5(),parseInt(t)||-1;},Kt.f6=function(t){Kt.Z5();var A=t.backingJsObject;if(A&&A.stack){var e=A.stack,r=A+m1;return e.substring(0,r.length)==r&&(e=e.substring(r.length)),e.split(m1);}return [];},Kt.g6=function(){return Error.stackTraceLimit>0?(d.Error.stackTraceLimit=Error.stackTraceLimit=64,true):"stack"in new Error();},I(447,1,{}),Kt.h6=function(){},I(312,447,{},Kt.h6),L.Nc=function(t){var A={},e=[];t[s1]=e;for(var r=arguments.callee.caller;r;){var P=Kt.d6(r);e.push(P);var i=":"+P,D=A[i];if(D){var u,o;for(u=0,o=D.length;u<o;u++)if(D[u]===r)return;}(D||(A[i]=[])).push(r),r=r.caller;}},L.Oc=function(t){var A,e,r,P;for(r=Kt.c6(t),e=h1.z5(r),P=B(Q.Mcb,Q1,90,e,0,1),A=0;A<e;A++)P[A]=new h.Sjb(h1.y5(r,A),null,-1);return P;},Kt.k6=function(t,A){var e,r,P,i,D,u,o,l,U;return l="",w.UAb(A).length==0?t.Pc(J1,L1,-1,-1):(U=h.ukb(A),h.ckb(w.UAb(U).substr(0,3),"at ")&&(U=(w.PAb(3,w.UAb(U).length+1),w.UAb(U).substr(3))),U=Kt.l6(U),D=w.UAb(U).indexOf("("),D==-1?(D=w.UAb(U).indexOf("@"),D==-1?(l=U,U=""):(l=h.ukb((w.PAb(D+1,w.UAb(U).length+1),w.UAb(U).substr(D+1))),U=h.ukb((w.OAb(0,D,w.UAb(U).length),w.UAb(U).substr(0,D))))):(e=w.UAb(U).indexOf(")",D),l=(w.OAb(D+1,e,w.UAb(U).length),w.UAb(U).substr(D+1,e-(D+1))),U=h.ukb((w.OAb(0,D,w.UAb(U).length),w.UAb(U).substr(0,D)))),D=h.jkb(U,X1(46)),D!=-1&&(U=(w.PAb(D+1,w.UAb(U).length+1),w.UAb(U).substr(D+1))),(w.UAb(U).length==0||h.ckb(U,"Anonymous function"))&&(U=L1),u=h.mkb(l,X1(58)),P=h.nkb(l,X1(58),u-1),o=-1,r=-1,i=J1,u!=-1&&P!=-1&&(i=(w.OAb(0,P,w.UAb(l).length),w.UAb(l).substr(0,P)),o=Kt.e6((w.OAb(P+1,u,w.UAb(l).length),w.UAb(l).substr(P+1,u-(P+1)))),r=Kt.e6((w.PAb(u+1,w.UAb(l).length+1),w.UAb(l).substr(u+1)))),t.Pc(i,U,o,r));},Kt.l6=function(t){return t.replace(/\[.*?\]/g,"");},I(448,447,{}),L.Nc=function(t){},L.Pc=function(t,A,e,r){return new h.Sjb(A,t+"@"+r,e<0?-1:e);},L.Oc=function(t){var A,e,r,P,i,D;if(P=Kt.f6(t),i=B(Q.Mcb,Q1,90,0,0,1),A=0,r=h1.z5(P),r==0)return i;for(D=Kt.k6(this,h1.y5(P,0)),h.ckb(D.d,L1)||(i[A++]=D),e=1;e<r;e++)i[A++]=Kt.k6(this,h1.y5(P,e));return i;},Kt.p6=function(){},I(313,448,{},Kt.p6),L.Pc=function(t,A,e,r){return new h.Sjb(A,t,-1);},Q.r6=function(t){return t;};function v(f,t){return Q.t6(f,t);}Q.t6=function(t,A){return h.zib(t,A);},Q.u6=function(t){return t.__elementTypeCategory$==null?10:t.__elementTypeCategory$;},Q.v6=function(t){return t.__elementTypeId$;};function T1(f,t,A,e,r,P){return Q.x6(f,t,A,e,r,0,P);}Q.x6=function(t,A,e,r,P,i,D){var u,o,l,U,y;if(U=P[i],l=i==D-1,u=l?r:0,y=Q.z6(u,U),r!=10&&j(v(t,D-i),A[i],e[i],u,y),!l)for(++i,o=0;o<U;++o)Q.B6(y,o,Q.x6(t,A,e,r,P,i,D));return y;};function B(f,t,A,e,r,P){var i;return i=Q.z6(r,e),r!=10&&j(v(f,P),t,A,r,i),i;}Q.z6=function(t,A){var e=new Array(A),r;switch(t){case 14:case 15:r=0;break;case 16:r=false;break;default:return e;}for(var P=0;P<A;++P)e[P]=r;return e;};function n8(f){return Array.isArray(f)&&f.yd===Yn;}Q.B6=function(t,A,e){return t[A]=e;};function a1(f,t,A){return Q.B6(f,t,A);}Q.D6=function(t,A){t.wd=A;},Q.E6=function(t,A){t.__elementTypeCategory$=A;},Q.F6=function(t,A){t.__elementTypeId$=A;};function j(f,t,A,e,r){return Q.D6(r,f),r.xd=t,r.yd=Yn,Q.F6(r,A),Q.E6(r,e),r;}Q.H6=function(t,A){return Q.u6(A)!=10&&j(h.Qc(A),A.xd,Q.v6(A),Q.u6(A),t),Q.r6(t);};function xu(f,t){return pA(f)?!!_D[t]:f.xd?!!f.xd[t]:gn(f)?!!XD[t]:Fe(f)?!!YD[t]:false;}function e8(f){return !Array.isArray(f)&&f.yd===Yn;}function Zt(f,t){return f!=null&&xu(f,t);}function Fe(f){return typeof f===o0;}function gn(f){return typeof f===$1;}function JD(f){return f!=null&&Q.K7(f)&&f.yd!==Yn;}function ZD(f,t){return Q.L7(f,t);}function pA(f){return typeof f===H0;}Q.K7=function(t){return typeof t===at||typeof t=="function";},Q.L7=function(t,A){return t&&A&&t instanceof A;};function j1(f){return f??null;}function pt(f){return Math.max(Math.min(f,Ot),-2147483648)|0;}var YD,XD,_D;Q.Afb=function(t){return t&&t.__java$exception;};function M1(f){var t;return Zt(f,27)?f:(t=Q.Afb(f),t||(t=new h1.f5(f),Kt.$5(t)),t);}function Lt(f){return f.backingJsObject;}function qu(f){var t;return f==null?false:(t=typeof f,h.ckb(t,o0)||h.ckb(t,$1)||h.ckb(t,H0)||f.$implements__java_io_Serializable||Array.isArray(f));}function Vu(f){var t;return t=typeof f,h.ckb(t,o0)||h.ckb(t,$1)||h.ckb(t,H0)?true:f!=null&&f.$implements__java_lang_Comparable;}h.Ihb=function(){h.Ihb=i1,ve=w.UAb(false),w.UAb(true);},h.Jhb=function(t){return w.VAb((w.IAb(t),t));},h.Khb=function(t,A){return Nu(w.VAb((w.IAb(t),t)),w.VAb((w.IAb(A),A)));},h.Lhb=function(t,A){return h.Khb(t,A);},h.Mhb=function(t,A){return w.IAb(t),j1(t)===j1(A);},h.Nhb=function(){return Q.ucb;},h.Ohb=function(t){return w.VAb((w.IAb(t),t))?1231:1237;};function Iu(f){return h.Ihb(),h.ckb(o0,typeof f);}function Nu(f,t){return h.Ihb(),f==t?0:f?1:-1;}h.Rhb=function(t,A){return h.Ihb(),pA(t)?h.Xjb(t,A):gn(t)?h.Sib(t,A):Fe(t)?h.Lhb(t,A):t.Sb(A);},YD={4:1,307:1,32:1};var ve;function Gu(f){return h.ckb(typeof f,H0)?true:f!=null&&f.$implements__java_lang_CharSequence;}h.mib=function(t){t.i=Ju++;},h.nib=function(t){t.n==null&&h.Dib(t);},h.oib=function(t){return h.nib(t),t.n;},h.pib=function(t){return (t.f&4)!=0;},h.qib=function(t){return (t.f&1)!=0;},h.rib=function(){h.mib(this),this.n=null,this.j=null,this.g=null,this.d=null,this.b=null,this.k=null,this.a=null;},h.tib=function(t){var A;return A=new h.rib(),A.n="Class$"+(t?"S"+t:""+A.i),A.b=A.n,A.j=A.n,A;};function N(f){var t;return t=h.tib(f),h.Hib(f,t),t;}function KA(f,t){var A;return A=h.tib(f),h.Hib(f,A),A.f=t?8:0,A;}function Ku(){var f;return f=h.tib(null),f.f=2,f;}function An(f){var t;return t=h.tib(f),t.k=f,t.f=1,t;}h.zib=function(t,A){var e=t.a=t.a||[];return e[A]||(e[A]=t.Vc(A));};function zu(f){if(f.$c())return null;var t=f.k;return me[t];}h.Dib=function(t){if(t.Zc()){var A=t.c;A.$c()?t.n="["+A.k:A.Zc()?t.n="["+A.Xc():t.n="[L"+A.Xc()+";",t.b=A.Wc()+"[]",t.j=A.Yc()+"[]";return;}var e=t.g,r=t.d;r=r.split("/"),t.n=h.Gib(".",[e,h.Gib("$",r)]),t.b=h.Gib(".",[e,h.Gib(".",r)]),t.j=r[r.length-1];},h.Gib=function(t,A){for(var e=0;!A[e]||A[e]=="";)e++;for(var r=A[e++];e<A.length;e++)!A[e]||A[e]==""||(r+=t+A[e]);return r;},h.Hib=function(t,A){if(t){A.k=t;var r=zu(A);if(!r){me[t]=[A];return;}r.wd=A;}},I(207,1,{},h.rib),L.Vc=function(t){var A;return A=new h.rib(),A.f=4,t>1?A.c=h.zib(this,t-1):A.c=this,A;},L.Wc=function(){return h.nib(this),this.b;},L.Xc=function(){return h.oib(this);},L.Yc=function(){return h.nib(this),this.j;},L.Zc=function(){return h.pib(this);},L.$c=function(){return h.qib(this);},L.pb=function(){return ((this.f&2)!=0?"interface ":(this.f&1)!=0?"":"class ")+(h.nib(this),this.n);},L.f=0,L.i=0;var Ju=1;h.Mib=function(t){return h.ckb($1,typeof t)||ZD(t,d.java.lang.Number$impl);},h.Nib=function(t){if(h.Lib==null&&(h.Lib=new RegExp("^\\s*[+-]?(NaN|Infinity|((\\d+\\.?\\d*)|(\\.\\d+))([eE][+-]?\\d+)?[dDfF]?)\\s*$")),!h.Lib.test(t))throw Lt(new h.Rjb(HA+t+'"'));return parseFloat(t);},h.Oib=function(t,A,e){var r,P,i,D,u;if(t==null)throw Lt(new h.Rjb(PA));for(i=w.UAb(t).length,D=i>0&&(w.PAb(0,w.UAb(t).length),w.UAb(t).charCodeAt(0)==45||(w.PAb(0,w.UAb(t).length),w.UAb(t).charCodeAt(0)==43))?1:0,r=D;r<i;r++)if(t2((w.PAb(r,w.UAb(t).length),w.UAb(t).charCodeAt(r)))==-1)throw Lt(new h.Rjb(HA+t+'"'));if(u=parseInt(t,10),P=u<A,isNaN(u))throw Lt(new h.Rjb(HA+t+'"'));if(P||u>e)throw Lt(new h.Rjb(HA+t+'"'));return u;},h.Pib=function(t){return gn(t)?h.Tib(t):t._c();},h.Qib=function(t){return gn(t)?h.Xib(t):t.ad();},I(120,1,{4:1,120:1}),h.Rib=function(t,A){return zA(w.WAb((w.IAb(t),t)),w.WAb((w.IAb(A),A)));},h.Sib=function(t,A){return h.Rib(t,A);},h.Tib=function(t){return w.WAb((w.IAb(t),t));},h.Uib=function(t,A){return w.IAb(t),j1(t)===j1(A);},h.Vib=function(){return Q.xcb;},h.Wib=function(t){return pt(w.WAb((w.IAb(t),t)));},h.Xib=function(t){return pt(w.WAb((w.IAb(t),t)));};function Zu(f){return h.ckb($1,typeof f);}function zA(f,t){return f<t?-1:f>t?1:f==t?f==0?zA(1/f,1/t):0:isNaN(f)?isNaN(t)?0:1:-1;}function Sr(f){return isNaN(f)?{l:0,m:0,h:524160}:w.QAb(f);}XD={4:1,32:1,308:1,120:1},h.GB=function(t){t.g=B(Q.Mcb,Q1,90,0,0,1);},h.HB=function(t){Kt.$5(t);},h.IB=function(t){return Kt._5(t);},h.JB=function(t){return t.j&&(t.backingJsObject!==xe&&t.jc(),t.g=null),t;},h.KB=function(t){return t.e;},h.LB=function(t,A){if(A instanceof Object)try{if(A.__java$exception=t,mt.userAgent.toLowerCase().indexOf("msie")!=-1&&ft.documentMode<9)return;var e=t;Object.defineProperties(A,{cause:{get:function(){var r=e.hc();return r&&r.fc();}},suppressed:{get:function(){return e.gc();}}});}catch{}},h.MB=function(t,A,e){var r,P,i,D,u;for(h.NB(t),P=(t.i==null&&(t.i=B(Q.Rcb,Q1,27,0,0,1)),t.i),i=0,D=P.length;i<D;++i)r=P[i],h.MB(r,A,"	"+e);u=t.e,u&&h.MB(u,A,e);},h.NB=function(t){var A,e,r;for(A=(t.g==null&&(t.g=h.IB(t)),t.g),e=0,r=A.length;e<r;++e);},h.OB=function(t,A){t.backingJsObject=A,h.LB(t,A);},h.PB=function(t){return h.QB(t,t.ic());},h.QB=function(t,A){var e;return e=h.oib(t.wd),A==null?e:e+": "+A;},h.RB=function(){h.GB(this),h.JB(this),this.jc();},h.SB=function(t){h.GB(this),this.f=t,h.JB(this),this.jc();},h.UB=function(t){if(!("stack"in t))try{throw t;}catch{}return t;};function Yu(f){var t;return f!=null&&(t=f.__java$exception,t)?t:ZD(f,TypeError)?new h.Ojb(f):new h.Z4(f);}I(27,1,{4:1,27:1}),L.ec=function(t){return new Error(t);},L.fc=function(){return this.backingJsObject;},L.gc=function(){var t,A,e;for(e=(this.i==null&&(this.i=B(Q.Rcb,Q1,27,0,0,1)),this.i),A=B(Q.Kcb,C1,1,e.length,5,1),t=0;t<e.length;t++)A[t]=e[t].backingJsObject;return A;},L.hc=function(){return h.KB(this);},L.ic=function(){return this.f;},L.jc=function(){h.OB(this,h.UB(this.ec(h.QB(this,this.f)))),h.HB(this);},L.pb=function(){return h.PB(this);},L.backingJsObject=xe,L.j=true,h.aC=function(){h.RB.call(this);},h.bC=function(t){h.GB(this),h.JB(this),this.backingJsObject=t,h.LB(this,t),this.f=t==null?PA:Xn(t);},h.cC=function(t){h.SB.call(this,t);},I(19,27,{4:1,19:1,27:1},h.cC),h.dC=function(){h.aC.call(this);},h.eC=function(t){h.bC.call(this,t);},h.fC=function(t){h.cC.call(this,t);},I(33,19,m0),h.Dhb=function(){h.dC.call(this);},h.Ehb=function(t){h.fC.call(this,t);},I(79,33,m0,h.Dhb,h.Ehb),h.Y4=function(){h.dC.call(this);},h.Z4=function(t){h.eC.call(this,t);},h.$4=function(t){h.fC.call(this,t);},I(129,33,m0,h.Z4),h.Njb=function(){h.Y4.call(this);},h.Ojb=function(t){h.Z4.call(this,t);},h.Pjb=function(t){h.$4.call(this,t);},I(70,129,m0,h.Njb,h.Ojb,h.Pjb),L.ec=function(t){return new TypeError(t);},h.Wjb=function(t,A){return w.PAb(A,w.UAb(t).length),w.UAb(t).charCodeAt(A);},h.Xjb=function(t,A){return h.Yjb(t,A);},h.Yjb=function(t,A){var e,r;return e=h.Tib(w.UAb(t)),r=h.Tib(w.UAb(A)),e==r?0:e<r?-1:1;},h.Zjb=function(t,A){return h.Yjb(w.UAb(t).toLowerCase(),w.UAb(A).toLowerCase());},h.$jb=function(t,A){return w.IAb(t),t+(""+(w.IAb(A),A));},h._jb=function(t){var A;return h.zkb(w.sAb(t,0,(A=t.length,w.pAb(),A)));},h.akb=function(t){return h.zkb(w.sAb(t,0,t.length));},h.bkb=function(t){var A;return A=w.UAb(Rn).length,h.ckb(w.UAb(t).substr(w.UAb(t).length-A,A),Rn);},h.ckb=function(t,A){return w.IAb(t),j1(t)===j1(A);},h.dkb=function(t,A){return w.IAb(t),A==null?false:h.ckb(t,A)?true:w.UAb(t).length==w.UAb(A).length&&h.ckb(w.UAb(t).toLowerCase(),w.UAb(A).toLowerCase());},h.ekb=function(t,A,e,r,P){w.OAb(A,e,w.UAb(t).length),w.OAb(P,P+(e-A),r.length),h.fkb(t,A,e,r,P);},h.fkb=function(t,A,e,r,P){for(;A<e;)r[P++]=h.Wjb(t,A++);},h.gkb=function(){return Q.Qcb;},h.hkb=function(t){var A,e;for(A=0,e=0;e<w.UAb(t).length;e++)A=(A<<5)-A+(w.PAb(e,w.UAb(t).length),w.UAb(t).charCodeAt(e))|0;return A;},h.ikb=function(t,A,e){return h.kkb(t,X1(A),e);},h.jkb=function(t,A){return w.UAb(t).indexOf(A);},h.kkb=function(t,A,e){return w.UAb(t).indexOf(A,e);};function Xu(f){return h.ckb(H0,typeof f);}h.mkb=function(t,A){return w.UAb(t).lastIndexOf(A);},h.nkb=function(t,A,e){return w.UAb(t).lastIndexOf(A,e);},h.okb=function(t,A){var e;return e=h.pkb(A,"([/\\\\\\.\\*\\+\\?\\|\\(\\)\\[\\]\\{\\}$^])","\\\\$1"),h.pkb(t,e,"");},h.pkb=function(t,A,e){return e=h.ykb(e),w.UAb(t).replace(new RegExp(A,"g"),e);},h.qkb=function(t,A){var e,r,P,i,D,u,o,l;for(e=new RegExp(A,"g"),o=B(Q.Qcb,yt,2,0,6,1),r=0,l=t,i=null;;)if(u=e.exec(l),u==null||l==""){o[r]=l;break;}else D=u.index,o[r]=(w.OAb(0,D,w.UAb(l).length),w.UAb(l).substr(0,D)),l=h.tkb(l,D+w.UAb(w.UAb(u)[0]).length,w.UAb(l).length),e.lastIndex=0,i==l&&(o[r]=(w.OAb(0,1,w.UAb(l).length),w.UAb(l).substr(0,1)),l=(w.PAb(1,w.UAb(l).length+1),w.UAb(l).substr(1))),i=l,++r;if(w.UAb(t).length>0){for(P=o.length;P>0&&o[P-1]=="";)--P;P<o.length&&(w.UAb(o).length=P);}return o;},h.rkb=function(t,A){return h.ckb(w.UAb(t).substr(0,w.UAb(A).length),A);},h.skb=function(t,A){return w.PAb(A,w.UAb(t).length+1),w.UAb(t).substr(A);},h.tkb=function(t,A,e){return w.OAb(A,e,w.UAb(t).length),w.UAb(t).substr(A,e-A);},h.ukb=function(t){var A,e,r;for(e=w.UAb(t).length,r=0;r<e&&(w.PAb(r,w.UAb(t).length),w.UAb(t).charCodeAt(r)<=32);)++r;for(A=e;A>r&&(w.PAb(A-1,w.UAb(t).length),w.UAb(t).charCodeAt(A-1)<=32);)--A;return r>0||A<e?(w.OAb(r,A,w.UAb(t).length),w.UAb(t).substr(r,A-r)):t;},h.vkb=function(t){return String.fromCharCode.apply(null,t);};function X1(f){var t,A;return f>=H1?(t=55296+(f-H1>>10&1023)&n1,A=56320+(f-H1&1023)&n1,String.fromCharCode(t)+(""+String.fromCharCode(A))):String.fromCharCode(f&n1);}h.xkb=function(t){var A,e,r,P,i;for(i=new s.Gwb(),e=t,r=0,P=e.length;r<P;++r)A=e[r],i.a?h.Kkb(i.a,i.b):i.a=new h.Okb(i.d),h.Jkb(i.a,A);return i.a?w.UAb(i.e).length==0?i.a.a:i.a.a+(""+i.e):i.c;},h.ykb=function(t){var A;for(A=0;0<=(A=w.UAb(t).indexOf("\\",A));)w.PAb(A+1,w.UAb(t).length),w.UAb(t).charCodeAt(A+1)==36?t=(w.OAb(0,A,w.UAb(t).length),w.UAb(t).substr(0,A)+"$"+h.skb(t,++A)):t=(w.OAb(0,A,w.UAb(t).length),w.UAb(t).substr(0,A)+(""+h.skb(t,++A)));return t;},h.zkb=function(t){return h.Akb(t,0,t.length);},h.Akb=function(t,A,e){var r,P,i,D;for(i=A+e,w.OAb(A,i,t.length),D="",P=A;P<i;)r=d.Math.min(P+Z0,i),D+=h.vkb(w.UAb(t).slice(P,r)),P=r;return D;},_D={4:1,206:1,32:1,2:1},h.Pkb=function(t){h.Ehb.call(this,t);},I(231,79,m0,h.Pkb),w.eAb=function(t){var A;return A=w.UAb(t).slice(),w.oAb(A,t);},w.fAb=function(t,A,e,r,P,i){var D,u,o,l,U;if(P!=0)for(j1(t)===j1(e)&&(t=w.UAb(t).slice(A,A+P),A=0),o=w.UAb(e),u=A,l=A+P;u<l;)D=d.Math.min(u+Z0,l),P=D-u,U=w.UAb(t).slice(u,D),w.UAb(U).splice(0,0,w.UAb(r),w.UAb(i?P:0)),Array.prototype.splice.apply(o,U),u=D,r+=P;},w.gAb=function(t,A){return w.oAb(new Array(A),t);},w.hAb=function(t,A,e){w.UAb(t).splice(A,0,e);},w.iAb=function(t,A,e){w.fAb(e,0,t,A,e.length,false);},w.jAb=function(t,A){w.UAb(t).push(A);},w.kAb=function(t,A){w.UAb(t).push(A);},w.lAb=function(t,A,e){w.UAb(t).splice(A,e);},w.mAb=function(t,A){w.UAb(t).length=A;},w.nAb=function(t,A){w.UAb(t).sort(A);},I(513,1,{}),w.xAb=function(){},w.yAb=function(t){switch(typeof t){case H0:return h.hkb(w.UAb(t));case $1:return h.Wib(w.UAb(t));case o0:return h.Ohb(w.UAb(t));default:return t==null?0:w.AAb(t);}},w.zAb=function(){return ++w.wAb;},w.AAb=function(t){return t.$H||(t.$H=w.zAb());},I(445,1,{},w.xAb),w.wAb=0,w.BAb=function(t){if(!t)throw Lt(new h.ajb());},w.CAb=function(t,A){if(!t)throw Lt(new h.bjb(A));},w.DAb=function(t,A,e){if(t>A)throw Lt(new h.bjb(te+t+I8+A));if(t<0||A>e)throw Lt(new h.Ghb(te+t+Xr+A+N8+e));},w.EAb=function(t){if(t<0)throw Lt(new h.Mjb("Negative array size: "+t));},w.FAb=function(t,A){if(t!=A)throw Lt(new s.oub());},w.GAb=function(t){if(!t)throw Lt(new s.Ivb());},w.HAb=function(t,A){if(t<0||t>=A)throw Lt(new h.Ehb(_r+t+Wr+A));},w.IAb=function(t){if(t==null)throw Lt(new h.Njb());return t;},w.JAb=function(t,A){if(t==null)throw Lt(new h.Pjb(A));},w.KAb=function(t,A){if(t<0||t>A)throw Lt(new h.Ehb(_r+t+Wr+A));},w.LAb=function(t,A,e){if(t<0||A>e)throw Lt(new h.Ehb(te+t+Xr+A+", size: "+e));if(t>A)throw Lt(new h.bjb(te+t+I8+A));},w.MAb=function(t){if(!t)throw Lt(new h.cjb());},w.NAb=function(t){if(!t)throw Lt(new h.djb("Unable to add element to queue"));},w.OAb=function(t,A,e){if(t<0||A>e||A<t)throw Lt(new h.Pkb(te+t+Xr+A+N8+e));},w.PAb=function(t,A){if(t<0||t>=A)throw Lt(new h.Pkb(_r+t+Wr+A));},w.QAb=function(t){var A,e;return A=new ArrayBuffer(8),w.UAb(new Float64Array(A))[0]=t,e=w.UAb(new Uint32Array(A)),w.XAb(e[0]|0,e[1]|0);},w.RAb=function(t,A){return t[A];},w.SAb=function(t){return t===void 0;},w.TAb=function(t){return t>>>0;},w.UAb=function(t){return t;},w.VAb=function(t){return t;},w.WAb=function(t){return t;},I(446,1,{}),Q.Kcb=N(1),Q.Tbb=N(0),Q.$bb=N(447),Q.Xbb=N(312),Q.Zbb=N(448),Q.Ybb=N(313),Q.ucb=N(307),Q.wcb=N(207),Q.Jcb=N(120),Q.xcb=N(308),Q.Rcb=N(27),Q.zcb=N(19),Q.Lcb=N(33),Q.Ccb=N(79),Q.Ecb=N(129),Q.Hcb=N(70),Q.Qcb=N(2),Q.Pcb=N(231),Q.Reb=N(445),Q.Seb=N(446),uA.Vc=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt;for(this.c=t.length,this.d=t[0].length,K=d.Math.min(this.c,this.d),this.e=B(Q.Q7,jt,6,d.Math.min(this.c+1,this.d),15,1),this.a=T1(Q.Q7,[Q1,jt],[15,6],15,[this.c,K],2),this.b=T1(Q.Q7,[Q1,jt],[15,6],15,[this.d,this.d],2),P=B(Q.Q7,jt,6,this.d,15,1),Vt=B(Q.Q7,jt,6,this.c,15,1),At=d.Math.min(this.c-1,this.d),ot=d.Math.max(0,d.Math.min(this.d-2,this.c)),O=0;O<d.Math.max(At,ot);O++){if(O<At){for(this.e[O]=0,y=O;y<this.c;y++)this.e[O]=uA.Wc(this.e[O],t[y][O]);if(this.e[O]!=0){for(t[O][O]<0&&(this.e[O]=-this.e[O]),U=O;U<this.c;U++)t[U][O]/=this.e[O];t[O][O]+=1;}this.e[O]=-this.e[O];}for(k=O+1;k<this.d;k++){if(O<At&this.e[O]!=0){for(ct=0,y=O;y<this.c;y++)ct+=t[y][O]*t[y][k];for(ct=-ct/t[O][O],U=O;U<this.c;U++)t[U][k]+=ct*t[U][O];}P[k]=t[O][k];}if(true&O<At)for(U=O;U<this.c;U++)this.a[U][O]=t[U][O];if(O<ot){for(P[O]=0,y=O+1;y<this.d;y++)P[O]=uA.Wc(P[O],P[y]);if(P[O]!=0){for(P[O+1]<0&&(P[O]=-P[O]),T=O+1;T<this.d;T++)P[T]/=P[O];P[O+1]+=1;}if(P[O]=-P[O],O+1<this.c&P[O]!=0){for(T=O+1;T<this.c;T++)Vt[T]=0;for(F=O+1;F<this.d;F++)for(c=O+1;c<this.c;c++)Vt[c]+=P[F]*t[c][F];for(S=O+1;S<this.d;S++)for(ct=-P[S]/P[O+1],c=O+1;c<this.c;c++)t[c][S]+=ct*Vt[c];}for(U=O+1;U<this.d;U++)this.b[U][O]=P[U];}}for(it=d.Math.min(this.d,this.c+1),At<this.d&&(this.e[At]=t[At][At]),this.c<it&&(this.e[it-1]=0),ot+1<it&&(P[ot]=t[ot][it-1]),P[it-1]=0,R=At;R<K;R++){for(U=0;U<this.c;U++)this.a[U][R]=0;this.a[R][R]=1;}for(x=At-1;x>=0;x--)if(this.e[x]!=0){for(g=x+1;g<K;g++){for(ct=0,y=x;y<this.c;y++)ct+=this.a[y][x]*this.a[y][g];for(ct=-ct/this.a[x][x],T=x;T<this.c;T++)this.a[T][g]+=ct*this.a[T][x];}for(c=x;c<this.c;c++)this.a[c][x]=-this.a[c][x];for(this.a[x][x]=1+this.a[x][x],U=0;U<x-1;U++)this.a[U][x]=0;}else {for(U=0;U<this.c;U++)this.a[U][x]=0;this.a[x][x]=1;}for(G=this.d-1;G>=0;G--){if(G<ot&P[G]!=0)for(g=G+1;g<K;g++){for(ct=0,y=G+1;y<this.d;y++)ct+=this.b[y][G]*this.b[y][g];for(ct=-ct/this.b[G+1][G],T=G+1;T<this.d;T++)this.b[T][g]+=ct*this.b[T][G];}for(U=0;U<this.d;U++)this.b[U][G]=0;this.b[G][G]=1;}for(lt=it-1,u=d.Math.pow(2,-52);it>0;){for(E=it-2;E>=-1&&E!=-1;E--)if(d.Math.abs(P[E])<=u*(d.Math.abs(this.e[E])+d.Math.abs(this.e[E+1]))){P[E]=0;break;}if(E==it-2)J=4;else {for(V=it-1;V>=E&&V!=E;V--)if(ct=(V!=it?d.Math.abs(P[V]):0)+(V!=E+1?d.Math.abs(P[V-1]):0),d.Math.abs(this.e[V])<=u*ct){this.e[V]=0;break;}V==E?J=3:V==it-1?J=1:(J=2,E=V);}switch(++E,J){case 1:for(o=P[it-2],P[it-2]=0,g=it-2;g>=E;g--)for(ct=uA.Wc(this.e[g],o),r=this.e[g]/ct,bt=o/ct,this.e[g]=ct,g!=E&&(o=-bt*P[g-1],P[g-1]=r*P[g-1]),U=0;U<this.d;U++)ct=r*this.b[U][g]+bt*this.b[U][it-1],this.b[U][it-1]=-bt*this.b[U][g]+r*this.b[U][it-1],this.b[U][g]=ct;break;case 2:for(o=P[E-1],P[E-1]=0,g=E;g<it;g++)for(ct=uA.Wc(this.e[g],o),r=this.e[g]/ct,bt=o/ct,this.e[g]=ct,o=-bt*P[g],P[g]=r*P[g],U=0;U<this.c;U++)ct=r*this.a[U][g]+bt*this.a[U][E-1],this.a[U][E-1]=-bt*this.a[U][g]+r*this.a[U][E-1],this.a[U][g]=ct;break;case 3:{for(ut=d.Math.max(d.Math.max(d.Math.max(d.Math.max(d.Math.abs(this.e[it-1]),d.Math.abs(this.e[it-2])),d.Math.abs(P[it-2])),d.Math.abs(this.e[E])),d.Math.abs(P[E])),st=this.e[it-1]/ut,wt=this.e[it-2]/ut,D=P[it-2]/ut,Qt=this.e[E]/ut,i=P[E]/ut,A=((wt+st)*(wt-st)+D*D)/2,e=st*D*(st*D),Dt=0,A!=0|e!=0&&(Dt=d.Math.sqrt(A*A+e),A<0&&(Dt=-Dt),Dt=e/(A+Dt)),o=(Qt+st)*(Qt-st)+Dt,l=Qt*i,g=E;g<it-1;g++){for(ct=uA.Wc(o,l),r=o/ct,bt=l/ct,g!=E&&(P[g-1]=ct),o=r*this.e[g]+bt*P[g],P[g]=r*P[g]-bt*this.e[g],l=bt*this.e[g+1],this.e[g+1]=r*this.e[g+1],y=0;y<this.d;y++)ct=r*this.b[y][g]+bt*this.b[y][g+1],this.b[y][g+1]=-bt*this.b[y][g]+r*this.b[y][g+1],this.b[y][g]=ct;if(ct=uA.Wc(o,l),r=o/ct,bt=l/ct,this.e[g]=ct,o=r*P[g]+bt*this.e[g+1],this.e[g+1]=-bt*P[g]+r*this.e[g+1],l=bt*P[g+1],P[g+1]=r*P[g+1],g<this.c-1)for(U=0;U<this.c;U++)ct=r*this.a[U][g]+bt*this.a[U][g+1],this.a[U][g+1]=-bt*this.a[U][g]+r*this.a[U][g+1],this.a[U][g]=ct;}P[it-2]=o;}break;case 4:{if(this.e[E]<=0)for(this.e[E]=this.e[E]<0?-this.e[E]:0,U=0;U<=lt;U++)this.b[U][E]=-this.b[U][E];for(;E<lt&&!(this.e[E]>=this.e[E+1]);){if(ct=this.e[E],this.e[E]=this.e[E+1],this.e[E+1]=ct,E<this.d-1)for(U=0;U<this.d;U++)ct=this.b[U][E+1],this.b[U][E+1]=this.b[U][E],this.b[U][E]=ct;if(E<this.c-1)for(U=0;U<this.c;U++)ct=this.a[U][E+1],this.a[U][E+1]=this.a[U][E],this.a[U][E]=ct;++E;}--it;}}}},uA.Wc=function(t,A){var e;return d.Math.abs(t)>d.Math.abs(A)?(e=A/t,e=d.Math.abs(t)*d.Math.sqrt(1+e*e)):A!=0?(e=t/A,e=d.Math.abs(A)*d.Math.sqrt(1+e*e)):e=0,e;},I(200,1,fn,uA.Vc),L.c=0,L.d=0,Q.U7=N(200),n._c=function(){n._c=i1,n.Xc=j(v(Q.S7,1),H,6,15,[0,WA,14286847,13402367,12779264,16758197,9474192,3166456,16715021,9494608,11789301,11230450,9109248,12560038,1578e4,16744448,16777008,2093087,8442339,9388244,4062976,15132390,12567239,10921643,9083335,10255047,14706227,15765664,5296208,13140019,8224944,12750735,6721423,12419299,16752896,10889513,6076625,7351984,Y0,9764863,9756896,7586505,5551541,3907230,2396047,687500,27013,12632256,16767375,10909043,6717568,10380213,13924864,9699476,4366e3,5707663,51456,7394559,16777159,14286791,13107143,10747847,9437127,6422471,4587463,3211207,2097095,65436,58997,54354,48952,43812,5096191,5089023,2200790,2522539,2516630,1528967,13684960,16765219,12105936,10900557,5724513,10375093,11230208,7688005,4358806,4325478,32e3,7384058,47871,41471,36863,33023,27647,5528818,7888099,9064419,10565332,11739092,11739066,11734438,12389767,13041766,13369433,13697103,14221381,14680120,15073326,15400998,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13158600,1334015,56540,15075850,15132160,56540,15075850,15461355,8553170,1016335,1016335,1334015,15132160,3289770,14456450,16422400,16422400,11819700,3289770,1016335]);},n.ad=function(t){t.v=new ht.pO();},n.bd=function(t){var A,e;(t.G&32)==0&&(e=n.ew(t.L),e!=null&&(t.w.a==0&&t.w.b==0&&(A=t.R.c*n.Ul(t.L),t.v=n.Zd(t),n.od(t,A),n.Ud(t,null,A,0)),t.Cb(pt(t.A)),t.L.O!=1&&n.Vd(t,448),t.tb(e,t.w.a,t.w.b+Dn*t.A)));},n.cd=function(t,A){return t==null?A:A==null?t:t+","+A;},n.dd=function(t,A){n.ui(A,t.R),n.xi(A,t.v),n.wi(A,t.w);},n.ed=function(t){var A;A=t.R.c*(t.r!=0?t.r:n.Ul(t.L)),t.Y=A*n.Zc,t.T=A*n.Yc,t.S=A*.38,t.W=A*.47,t.X=pt(A*t.J*n.$c+.5),t.V=A*.12,t.Z=A*.4,t.A=A*.5+.5;},n.fd=function(t){var A,e;for(t=z(t,EA),e=0;e<(M.tK(),M.sK).length;e++)if(S1(t,M.sK[e]))return M.rK[e];return A=new h.Okb("R"),Rt(z(t,lA),0)&&(A.a+="0"),Rt(z(t,OA),0)&&(A.a+="3"),Rt(z(t,xA),0)&&(A.a+="4"),Rt(z(t,qA),0)&&(A.a+="5"),Rt(z(t,VA),0)&&(A.a+="6"),Rt(z(t,IA),0)&&(A.a+="7"),Rt(z(t,un),0)&&(A.a+="8"),A.a;},n.gd=function(t,A,e,r){var P,i;P=new n.je(),i=new n.je(),P.a=A.a,P.c=A.c,P.b=(A.a+A.b)/2,P.d=(A.c+A.d)/2,i.a=P.b,i.c=P.d,i.b=A.b,i.d=A.d,n.Rd(t,P)&&(n.Vd(t,t.p[e]),t.qb(P)),n.Rd(t,i)&&(n.Vd(t,t.p[r]),t.qb(i)),n.Vd(t,t.Q);},n.hd=function(t,A,e,r){var P,i,D,u,o;u=(A.b-A.a)/10,o=(A.d-A.c)/10,P=new n.je(),n.Dm(t.L,n.Xo(t.L,e,r))?(i=-3,D=-3):(i=t.p[e],D=t.p[r]),n.Vd(t,i),P.a=A.a,P.c=A.c,P.b=A.a+u*2,P.d=A.c+o*2,t.qb(P),P.a=A.a+u*4,P.c=A.c+o*4,P.b=A.a+u*5,P.d=A.c+o*5,t.qb(P),n.Vd(t,D),P.a=A.a+u*5,P.c=A.c+o*5,P.b=A.a+u*6,P.d=A.c+o*6,t.qb(P),P.a=A.a+u*8,P.c=A.c+o*8,P.b=A.b,P.d=A.d,t.qb(P),n.Vd(t,t.Q);},n.jd=function(t,A,e){t.ub(A-t.V/2,e-t.V/2,t.V);},n.kd=function(t,A,e,r){n.Dm(t.L,n.Xo(t.L,e,r))?(n.Vd(t,-3),t.qb(A),n.Vd(t,t.Q)):t.p[e]!=t.p[r]?n.gd(t,A,e,r):t.p[e]!=0?(n.Vd(t,t.p[e]),t.qb(A),n.Vd(t,t.Q)):t.qb(A);},n.ld=function(t,A,e,r,P,i,D){D.a=A.a+P,D.c=A.c+i,D.b=A.b+P,D.d=A.d+i,n.kd(t,D,e,r);},n.md=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g;for(y=A.b-A.a,b=A.d-A.c,o=d.Math.sqrt(y*y+b*b),l=2*zt(w1(d.Math.round(o/(4*t.Y)))),T=y/(l-1),g=b/(l-1),n.Dm(t.L,n.Xo(t.L,e,r))?(P=-3,i=-3):(P=t.p[e],i=t.p[r]),U=A.a-t.Y/2,c=A.c-t.Y/2,n.Vd(t,P),u=0;u<(l/2|0);u++)t.ub(U,c,t.Y),U+=T,c+=g;for(n.Vd(t,i),D=0;D<(l/2|0);D++)t.ub(U,c,t.Y),U+=T,c+=g;n.Vd(t,t.Q);},n.nd=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k,F;o=(A.c-A.d)/9,c=(A.b-A.a)/9,l=A.b+o,b=A.d+c,U=A.b-o,g=A.d-c,y=(A.a+l)/2,k=(A.c+b)/2,T=(A.a+U)/2,F=(A.c+g)/2,D=new ht.jO(3),u=new ht.jO(4),ht.eO(D,A.a,A.c),ht.eO(D,y,k),ht.eO(D,T,F),ht.eO(u,T,F),ht.eO(u,y,k),ht.eO(u,l,b),ht.eO(u,U,g),n.Dm(t.L,n.Xo(t.L,e,r))?(P=-3,i=-3):(P=t.p[e],i=t.p[r],t.L.O!=1&&(i=n.qd(t,e),P==n.Cl(t.L,e)&&(P=i))),n.Vd(t,P),t.sb(D),n.Vd(t,i),t.sb(u),n.Vd(t,t.Q);},n.od=function(t,A){var e,r;for(r=0;r<t.$.a.length;r++)t.v=ht.oO(t.v,s.xj(t.$,r));n.pd(t,A),e=.1*A,t.v.c-=e,t.v.d-=e,t.v.b+=2*e,t.v.a+=2*e;},n.pd=function(t,A){var e,r,P,i,D,u,o;for(P=B(Q.xfb,Tt,6,t.L.q,16,1),r=0;r<t.L.r;r++)n.Bm(t.L,r)&&(P[n.Zl(t.L,0,r)]=true,P[n.Zl(t.L,1,r)]=true);for(D=new ht.pO(),e=0;e<t.L.q;e++)i=n.Jm(t.L,e)?A*.47:P[e]?A*.38:0,i!=0&&(u=n.Ai(t.R,n.Ql(t.L,e)),o=n.Bi(t.R,n.Rl(t.L,e)),ht.nO(D,u-i,o-i,i*2,i*2),t.v=ht.oO(t.v,D));},n.qd=function(t,A){var e,r;if((t.G&4224)!=0||(r=n.rd(t,A),r==-1&&(e=n.Fo(t.L,A),e!=-1&&(A=e,r=n.rd(t,A))),r==-1))return t.p[A];switch(r&255){case 1:return 384;case 2:return 64;default:return 448;}},n.rd=function(t,A){var e,r,P;return P=-1,r=-1,(t.G&128)!=0||(n.zm(t.L,A)&&(P=n.Hl(t.L,A),r=n.Gl(t.L,A)),e=n.Ho(t.L,A),e!=-1&&(P=n.cm(t.L,e),r=n.bm(t.L,e)),P!=-1&&P!=0&&(P|=r<<8)),P;},n.sd=function(t){var A,e,r,P;for(t.Ab(2*t.S),P=new n.je(),r=0;r<t.L.r;r++)A=n.Zl(t.L,0,r),e=n.Zl(t.L,1,r),n.Bm(t.L,r)&&(P.a=n.Ai(t.R,n.Ql(t.L,A)),P.c=n.Bi(t.R,n.Rl(t.L,A)),P.b=n.Ai(t.R,n.Ql(t.L,e)),P.d=n.Bi(t.R,n.Rl(t.L,e)),n.Vd(t,-2),t.qb(P));},n.td=function(t){var A,e,r,P,i,D;if(t.L.K){for(D=t.W,n.Vd(t,-7),A=0;A<t.L.f;A++)n.Jm(t.L,A)&&t.ub(n.Ai(t.R,n.Ql(t.L,A))-D,n.Bi(t.R,n.Rl(t.L,A))-D,2*D);for(t.Ab(2*t.W),i=new n.je(),P=0;P<t.L.r;P++)e=n.Zl(t.L,0,P),r=n.Zl(t.L,1,P),Rt(z(z(n.Ol(t.L,e),n.Ol(t.L,r)),i0),0)&&(i.a=n.Ai(t.R,n.Ql(t.L,e)),i.c=n.Bi(t.R,n.Rl(t.L,e)),i.b=n.Ai(t.R,n.Ql(t.L,r)),i.d=n.Bi(t.R,n.Rl(t.L,r)),t.qb(i));}},n.ud=function(t){var A,e,r,P;if(t.L.K){if(n.Vd(t,320),(t.G&8)!=0)for(A=0;A<t.L.f;A++)Rt(z(n.Ol(t.L,A),-536870913),0)&&t.ub(n.Ai(t.R,n.Ql(t.L,A))-t.Z/2,n.Bi(t.R,n.Rl(t.L,A))-t.Z/2,t.Z);for(P=0;P<t.L.g;P++)n.hm(t.L,P)!=0&&(e=n.Zl(t.L,0,P),r=n.Zl(t.L,1,P),t.ub((n.Ai(t.R,n.Ql(t.L,e))+n.Ai(t.R,n.Ql(t.L,r))-t.Z)/2,(n.Bi(t.R,n.Rl(t.L,e))+n.Bi(t.R,n.Rl(t.L,r))-t.Z)/2,t.Z));}},n.vd=function(t){t.J=1,t.R=new n.Ci(),t.$=new s.Mj(),t.U=new s.Mj(),t.s=B(Q.xfb,Tt,6,t.L.q,16,1),t.w=new ht.bO(),t.Q=0,t.C=-1,n._d(t);},n.wd=function(t,A){var e;if(n.Lo(t.L,A)==0)return  false;for(e=0;e<n.Lo(t.L,A);e++)if(!n.Dm(t.L,n._o(t.L,A,e)))return  false;return  true;},n.xd=function(t){var A;for(t.q=B(Q.xfb,Tt,6,t.L.q,16,1),A=0;A<t.L.r;A++)t.q[n.Zl(t.L,0,A)]=true,t.q[n.Zl(t.L,1,A)]=true;},n.yd=function(t,A){var e;if(n.$o(t.L,A)!=2)return  false;for(e=0;e<2;e++)if(n.ap(t.L,A,e)!=2)return  false;return  true;},n.zd=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b;for(T=false,P.a=0,P.b=0,r>0?i=$r:i=ar,b=n.Yl(t.L,A,e),U=0;U<n.$o(t.L,A);U++)D=n._o(t.L,A,U),u=b,n.Zl(t.L,0,D)==A?y=n.Zl(t.L,1,D):y=n.Zl(t.L,0,D),y!=e&&(c=n.Yl(t.L,A,y),u<c&&(u+=N1),o=u-c,r>0?(o<t1&&(T=true),o>$r&&(o=$r),o<.523598776&&(o=.523598776),o<=i&&(i=o,l=t.T*d.Math.tan(i-F0)/2,P.a=-(l*d.Math.sin(u)),P.b=-(l*d.Math.cos(u)))):(o>=t1&&(T=true),o<ar&&(o=ar),o>5.759586531&&(o=5.759586531),o>=i&&(i=o,l=t.T*d.Math.tan(4.712388981-i)/2,P.a=-(l*d.Math.sin(u)),P.b=-(l*d.Math.cos(u)))));return T;},n.Ad=function(t,A,e,r){var P;if(A==0){e<0?r.a=t.T:r.a=-t.T,r.b=0;return;}P=d.Math.atan(e/A),A<0&&(P+=t1),r.a=-(t.T*d.Math.sin(P)),r.b=t.T*d.Math.cos(P);},n.Bd=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g;P=new n.je(),o=new n.je(),U=new ht.bO(),l=new ht.bO(),D=n.Zl(t.L,0,e),u=n.Zl(t.L,1,e),r&&(T=A.a,A.a=A.b,A.b=T,T=A.c,A.c=A.d,A.d=T,c=D,D=u,u=c),n.Rd(t,A)&&(n.gq(t.L,e)?(P.a=A.a,P.c=A.c,P.b=A.b,P.d=A.d,y=r?-n.Dp(t.L,e):n.Dp(t.L,e),y==0&&(y=1),n.Ad(t,A.b-A.a,A.d-A.c,U),y>0?(o.a=A.a+U.a,o.c=A.c+U.b,o.b=A.b+U.a,o.d=A.d+U.b,(n.zd(t,D,u,1,l)||n.$o(t.L,D)>1)&&(o.a+=l.a+U.b,o.c+=l.b-U.a)):(o.a=A.a-U.a,o.c=A.c-U.b,o.b=A.b-U.a,o.d=A.d-U.b,(n.zd(t,D,u,-1,l)||n.$o(t.L,D)>1)&&(o.a+=l.a+U.b,o.c+=l.b-U.a)),n.im(t.L,e)==386&&n.Qd(P,o),n.Rd(t,P)&&n.kd(t,P,D,u),n.im(t.L,e)==8?n.Rd(t,o)&&n.hd(t,o,D,u):n.Rd(t,o)&&n.kd(t,o,D,u)):(n.Ad(t,A.b-A.a,A.d-A.c,U),b=U.a/2,g=U.b/2,i=false,P.a=A.a+b,P.c=A.c+g,P.b=A.b+b,P.d=A.d+g,n.$o(t.L,D)>1&&(n.zd(t,D,u,1,l)?(P.a+=l.a,P.c+=l.b,n.$o(t.L,D)==2&&(l.a!=0||l.b!=0)&&(P.a+=U.b,P.c-=U.a)):t.o[D]=new ht.cO(P.a,P.c)),o.a=A.a-b,o.c=A.c-g,o.b=A.b-b,o.d=A.d-g,n.$o(t.L,D)>1&&(n.zd(t,D,u,0,l)?(o.a+=l.a,o.c+=l.b,n.$o(t.L,D)==2&&(l.a!=0||l.b!=0)&&(o.a+=U.b,o.c-=U.a)):(t.o[D]=new ht.cO(o.a,o.c),i=true)),n.im(t.L,e)==386&&n.Qd(P,o),n.im(t.L,e)==8?i?(n.hd(t,P,D,u),n.kd(t,o,D,u)):(n.kd(t,P,D,u),n.hd(t,o,D,u)):(n.kd(t,P,D,u),n.kd(t,o,D,u))));},n.Cd=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b;for(U=t.Q,y=t.F,(t.G&N0)!=0&&(t.Q=-6,t.F=qe,n.Vd(t,1)),t.o=B(Q.Zab,C1,55,t.L.q,0,1),o=0;o<t.L.r;o++)(n.im(t.L,o)==2||n.im(t.L,o)==386||n.im(t.L,o)==8)&&n.Fd(t,o);for(l=0;l<t.L.r;l++)n.im(t.L,l)!=2&&n.im(t.L,l)!=386&&n.im(t.L,l)!=8&&n.Fd(t,l);if((t.G&64)==0)for(u=0;u<t.L.r;u++)n.am(t.L,u)!=0&&(P=null,n.am(t.L,u)==1||n.am(t.L,u)==2?(n.fm(t.L,u)==2||n.cm(t.L,u)==0||A[n.cm(t.L,u)][n.bm(t.L,u)]>1)&&(n.am(t.L,u)==1?P=n.fm(t.L,u)==2?"E":n.Fm(t.L,u)?"p":"P":P=n.fm(t.L,u)==2?"Z":n.Fm(t.L,u)?"m":"M"):P="?",P!=null&&(t.Cb((t.X*2+1)/3|0),n.Vd(t,n.Dm(t.L,u)?-3:t.L.O==1||(t.G&Z1)!=0?t.Q:448),e=n.Zl(t.L,0,u),r=n.Zl(t.L,1,u),c=(n.Ai(t.R,n.Ql(t.L,e))+n.Ai(t.R,n.Ql(t.L,r)))/2,b=(n.Bi(t.R,n.Rl(t.L,e))+n.Bi(t.R,n.Rl(t.L,r)))/2,i=(n.Ai(t.R,n.Ql(t.L,e))-n.Ai(t.R,n.Ql(t.L,r)))/3,D=(n.Bi(t.R,n.Rl(t.L,e))-n.Bi(t.R,n.Rl(t.L,r)))/3,n.Id(t,c+D,b-i,P,true),n.Vd(t,t.Q),t.Cb(t.X)));if((t.G&4)!=0){for(t.Cb((t.X*2+1)/3|0),n.Vd(t,384),u=0;u<t.L.r;u++)e=n.Zl(t.L,0,u),r=n.Zl(t.L,1,u),T=n.Yp(t.L,u)?"d":n.Up(t.L,u)?"a":"",c=(n.Ai(t.R,n.Ql(t.L,e))+n.Ai(t.R,n.Ql(t.L,r)))/2,b=(n.Bi(t.R,n.Rl(t.L,e))+n.Bi(t.R,n.Rl(t.L,r)))/2,n.Id(t,c,b,T+(""+u),true);n.Vd(t,t.Q),t.Cb(t.X);}(t.G&N0)!=0&&(t.Q=U,t.F=y);},n.Dd=function(t){var A,e;for(e=new s.trb(t.U);e.a<e.c.a.length;)A=s.srb(e),n.Vd(t,A.a),n.jd(t,A.b,A.c);n.Vd(t,t.Q);},n.Ed=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt,Gt,dt,Bt;if(t.K||t.yb(A,n.Il(t.L,A),n.Ai(t.R,n.Ql(t.L,A)),n.Bi(t.R,n.Rl(t.L,A))),st=null,n.Bl(t.L,A)!=0&&(Jt=d.Math.abs(n.Bl(t.L,A))==1?"":""+d.Math.abs(n.Bl(t.L,A)),st=n.Bl(t.L,A)<0?Jt+"-":Jt+"+"),K=null,wt=n.Ol(t.L,A),nt(wt,0)!=0&&(Rt(z(wt,Ae),0)&&(K=K==null?"*":K+",*"),Rt(z(wt,ne),0)&&(K=K==null?"!*":K+",!*"),Rt(z(wt,ee),0)?K=K==null?"ha":K+",ha":Rt(z(wt,2),0)?K=K==null?"a":K+",a":Rt(z(wt,4),0)&&(K=K==null?"!a":K+",!a"),Rt(z(wt,Z1),0)&&(K=K==null?"s":K+",s"),Rt(z(wt,X0),0)&&(x=z(wt,X0),nt(x,1792)==0?K=K==null?"h0":K+",h0":nt(x,1664)==0?K=K==null?"h1":K+",h1":nt(x,1408)==0?K=K==null?"h2":K+",h2":nt(x,128)==0?K=K==null?"h>0":K+",h>0":nt(x,384)==0?K=K==null?"h>1":K+",h>1":nt(x,896)==0?K=K==null?"h>2":K+",h>2":nt(x,f0)==0?K=K==null?"h<3":K+",h<3":nt(x,1536)==0?K=K==null?"h<2":K+",h<2":nt(x,1152)==0&&(K=K==null?"h1-2":K+",h1-2")),Rt(z(wt,mA),0)&&(o=z(wt,mA),nt(o,re)==0?K=K==null?"c0":K+",c0":nt(o,on)==0?K=K==null?"c+":K+",c+":nt(o,Qn)==0&&(K=K==null?"c-":K+",c-")),Rt(z(wt,$A),0)&&(bt=z(wt,$A),nt(bt,98304)==0?K=K==null?"pi0":K+",pi0":nt(bt,81920)==0?K=K==null?"pi1":K+",pi1":nt(bt,49152)==0?K=K==null?"pi2":K+",pi2":nt(bt,N0)==0&&(K=K==null?"pi>0":K+",pi>0")),Rt(z(wt,aA),0)&&(Qt=z(wt,aA),nt(Qt,Pe)==0?K=K==null?"n1":K+",n1":nt(Qt,pn)==0?K=K==null?"n2":K+",n2":nt(Qt,Cn)==0?K=K==null?"n3":K+",n3":nt(Qt,ie)==0?K=K==null?"n<3":K+",n<3":nt(Qt,j0)==0?K=K==null?"n<4":K+",n<4":nt(Qt,tn)==0?K=K==null?"n>1":K+",n>1":nt(Qt,iA)==0?K=K==null?"n>2":K+",n>2":nt(Qt,ln)==0?K=K==null?"n>3":K+",n>3":nt(Qt,3276800)==0?K=K==null?"n1-2":K+",n1-2":nt(Qt,2228224)==0?K=K==null?"n1-3":K+",n1-3":nt(Qt,2490368)==0&&(K=K==null?"n2-3":K+",n2-3")),Rt(z(wt,Un),0)&&(b=z(wt,Un),nt(b,Ve)==0?K=K==null?"e0":K+",e0":nt(b,Ie)==0?K=K==null?"e1":K+",e1":nt(b,Ne)==0?K=K==null?"e2":K+",e2":nt(b,Ge)==0?K=K==null?"e3":K+",e3":nt(b,tP)==0?K=K==null?"e<2":K+",e<2":nt(b,AP)==0?K=K==null?"e<3":K+",e<3":nt(b,fe)==0?K=K==null?"e<4":K+",e<4":nt(b,Mn)==0?K=K==null?"e>0":K+",e>0":nt(b,Ke)==0?K=K==null?"e>1":K+",e>1":nt(b,ze)==0?K=K==null?"e>2":K+",e>2":nt(b,De)==0?K=K==null?"e>3":K+",e>3":nt(b,nP)==0?K=K==null?"e1-2":K+",e1-2":nt(b,eP)==0?K=K==null?"e1-3":K+",e1-3":nt(b,rP)==0&&(K=K==null?"e2-3":K+",e2-3")),Rt(z(wt,120),0)&&(ct=z(wt,120),nt(ct,112)==0?K=K==null?"!r":K+",!r":nt(ct,8)==0?K=K==null?"r":K+",r":nt(ct,96)==0?K=K==null?"rb<3":K+",rb<3":nt(ct,104)==0?K=K==null?"rb2":K+",rb2":nt(ct,88)==0?K=K==null?"rb3":K+",rb3":nt(ct,56)==0&&(K=K==null?"rb4":K+",rb4")),Rt(z(wt,FA),0)&&(K=K==null?"r"+s0(A0(z(wt,FA),22)):K+","+("r"+s0(A0(z(wt,FA),22)))),Rt(z(wt,EA),0)&&(K=n.cd(K,n.fd(wt))),Rt(z(wt,NA),0)&&(K=K==null?"f":K+",f")),n.Ml(t.L,A)!=0&&(K=n.cd(K,""+n.Ml(t.L,A))),Ht=0,n.Pl(t.L,A)!=0)switch(n.Pl(t.L,A)){case 16:st=st==null?"|":st+",|";break;case 32:Ht=1;break;case 48:Ht=2;}if(y=null,(t.G&64)==0){if(n.wm(t.L,A))y="?";else if(n.Al(t.L,A)!=0&&(n.Hl(t.L,A)==0||e==null||e[n.Hl(t.L,A)][n.Gl(t.L,A)]>1))if(n.$o(t.L,A)==2)switch(n.Al(t.L,A)){case 2:y=n.ym(t.L,A)?"p":"P";break;case 1:y=n.ym(t.L,A)?"m":"M";break;default:y="*";}else switch(n.Al(t.L,A)){case 1:y=n.ym(t.L,A)?"r":"R";break;case 2:y=n.ym(t.L,A)?"s":"S";break;default:y="*";}}if((t.G&768)!=0&&(y=n.cd(y,""+n.mw(t.L,A))),ut=null,(t.G&16)!=0&&n.Ll(t.L,A)!=0&&(ut=""+n.Ll(t.L,A)),k=null,n.Ip(t.L,A)!=-1&&(g=n.rd(t,A),g!=-1&&(k=g==0?"abs":((g&255)==1?"&":"or")+(1+(g>>8)))),G=0,(t.G&U0)==0&&(t.L.K?Rt(z(n.Ol(t.L,A),B1),0)&&(G=n.op(t.L,A)):(n.Tl(t.L,A)!=6||n.Ml(t.L,A)!=0||!t.q[A]||n.Pl(t.L,A)!=0)&&(G=n.op(t.L,A))),ot=false,At=false,i=(t.G&H1)!=0?null:n.El(t.L,A),i!=null&&h.ckb(w.UAb(i).substr(0,1),"]")&&(ot=K==null,At=true,K=n.cd((w.PAb(1,w.UAb(i).length+1),w.UAb(i).substr(1)),K),i=null),i!=null?G=0:n.Jl(t.L,A)!=null?(P=Rt(z(n.Ol(t.L,A),1),0)?"[!":"[",i=P+n.Kl(t.L,A)+"]",w.UAb(i).length>5&&(i=P+n.Jl(t.L,A).length+"]"),Rt(z(n.Ol(t.L,A),B1),0)&&(G=-1)):Rt(z(n.Ol(t.L,A),1),0)?(i="?",Rt(z(n.Ol(t.L,A),B1),0)&&(G=-1)):(n.Tl(t.L,A)!=6||st!=null||K!=null&&(!ot||(t.G&UA)==0)||G>0||!t.q[A])&&(i=n.Il(t.L,A)),lt=0,!n.Pm(t.L,A)&n.Jm(t.L,A)&&n.Vd(t,-8),i!=null?(lt=t.vb(i),n.Id(t,n.Ai(t.R,n.Ql(t.L,A)),n.Bi(t.R,n.Rl(t.L,A)),i,true),t.s[A]=true):n.yd(t,A)&&n.Hd(t,n.Ai(t.R,n.Ql(t.L,A)),n.Bi(t.R,n.Rl(t.L,A)),A),st!=null&&(t.Cb((t.X*2+1)/3|0),Nt=n.Ai(t.R,n.Ql(t.L,A))+((lt+t.vb(st))/2+1),dt=n.Bi(t.R,n.Rl(t.L,A))-((t.wb()*4-4)/8|0),n.Id(t,Nt,dt,st,true),t.Cb(t.X)),(t.G&2)!=0&&(K=""+A),K!=null&&(At?t.Cb((t.X*5+1)/6|0):t.Cb((t.X*2+1)/3|0),Nt=n.Ai(t.R,n.Ql(t.L,A))-(lt+t.vb(K))/2,dt=n.Bi(t.R,n.Rl(t.L,A))-((t.wb()*4-4)/8|0),n.Id(t,Nt,dt,K,true),t.Cb(t.X)),y!=null&&(t.Cb((t.X*2+1)/3|0),Nt=n.Ai(t.R,n.Ql(t.L,A))-(lt+t.vb(y))/2,dt=n.Bi(t.R,n.Rl(t.L,A))+((t.wb()*4+4)/8|0),Et=t.C,t.L.O!=1&&(t.G&Z1)==0&&n.Vd(t,448),n.Id(t,Nt,dt,y,false),n.Vd(t,Et),t.Cb(t.X)),ut!=null&&(t.Cb((t.X*2+1)/3|0),Nt=n.Ai(t.R,n.Ql(t.L,A))+((lt+t.vb(ut))/2+1),dt=n.Bi(t.R,n.Rl(t.L,A))+((t.wb()*4+4)/8|0),Et=t.C,n.Vd(t,n.Am(t.L,A)?384:448),n.Id(t,Nt,dt,ut,true),n.Vd(t,Et),t.Cb(t.X)),k!=null&&(r=n.Md(t,A),t.Cb((t.X*2+1)/3|0),Nt=n.Ai(t.R,n.Ql(t.L,A))+.7*t.wb()*d.Math.sin(r),dt=n.Bi(t.R,n.Rl(t.L,A))+.7*t.wb()*d.Math.cos(r),Et=t.C,!t.K&&t.L.O!=1&&n.Vd(t,n.qd(t,A)),n.Id(t,Nt,dt,k,false),n.Vd(t,Et),t.Cb(t.X)),G==0&&Ht==0){t.C==-8&&n.Vd(t,-9);return;}for(E=B(Q.Q7,jt,6,4,15,1),V=0;V<n.Mo(t.L,A);V++)for(u=n._o(t.L,A,V),it=0;it<2;it++)n.Zl(t.L,it,u)==A&&(Vt=n.Yl(t.L,n.Zl(t.L,it,u),n.Zl(t.L,1-it,u)),Vt<ue?(E[0]-=Vt+F0,E[3]+=Vt+t1):Vt<0?(E[2]+=Vt+F0,E[3]-=Vt):Vt<F0?(E[1]+=Vt,E[2]+=F0-Vt):(E[0]+=Vt-F0,E[1]+=t1-Vt));if(n.$o(t.L,A)==0?n.Hm(t.L,A)?E[3]-=.2:E[1]-=.2:E[1]-=.1,(st!=null||ut!=null)&&(E[1]+=10),(K!=null||y!=null)&&(E[3]+=10),S="",G!=0&&(O=t.vb("H"),R=0,F=t.wb(),G==-1?(S="n",t.Cb((t.X*2+1)/3|0),R=t.vb(S)):G>1&&(S=""+G,t.Cb((t.X*2+1)/3|0),R=t.vb(S)),E[1]<.6||E[3]<.6?(U=n.Bi(t.R,n.Rl(t.L,A)),E[1]<=E[3]?(E[1]+=10,l=n.Ai(t.R,n.Ql(t.L,A))+(lt+O)/2):(E[3]+=10,l=n.Ai(t.R,n.Ql(t.L,A))-(lt+O)/2-R)):(l=n.Ai(t.R,n.Ql(t.L,A)),E[0]<E[2]?(E[0]+=10,U=n.Bi(t.R,n.Rl(t.L,A))-F):(E[2]+=10,U=n.Bi(t.R,n.Rl(t.L,A))+F)),R>0&&(Nt=l+(O+R)/2,dt=U+((t.wb()*4+4)/8|0),n.Id(t,Nt,dt,S,true),t.Cb(t.X)),n.Id(t,l,U,"H",true)),D=0,Ht!=0){for(Dt=50,T=0,J=0;J<4;J++)c=J>1?J-2:J+2,E[J]<Dt?(D=J,Dt=E[J],T=E[c]):E[J]==Dt&&E[c]>T&&(D=J,T=E[c]);switch(D){case 0:l=n.Ai(t.R,n.Ql(t.L,A)),U=n.Bi(t.R,n.Rl(t.L,A))-t.V-lt/2;break;case 1:l=n.Ai(t.R,n.Ql(t.L,A))+t.V+lt/2,U=n.Bi(t.R,n.Rl(t.L,A));break;case 2:l=n.Ai(t.R,n.Ql(t.L,A)),U=n.Bi(t.R,n.Rl(t.L,A))+t.V+lt/2;break;default:l=n.Ai(t.R,n.Ql(t.L,A))-t.V-lt/2,U=n.Bi(t.R,n.Rl(t.L,A));}if(Ht==1)s.sj(t.$,new ht.qO(l-t.V,U-t.V,2*t.V,2*t.V)),t.K||s.sj(t.U,new n.ie(l,U,n.wd(t,A)?-3:t.p[A]));else {switch(D){case 2:case 0:Gt=2*t.V,Bt=0,l-=t.V;break;case 1:Gt=0,Bt=2*t.V,U-=t.V;break;default:Gt=0,Bt=2*t.V,U-=t.V;}s.sj(t.$,new ht.qO(l-t.V,U-t.V,2*t.V,2*t.V)),t.K||s.sj(t.U,new n.ie(l,U,n.wd(t,A)?-3:t.p[A])),s.sj(t.$,new ht.qO(l+Gt-t.V,U+Bt-t.V,2*t.V,2*t.V)),t.K||s.sj(t.U,new n.ie(l+Gt,U+Bt,n.wd(t,A)?-3:t.p[A]));}}t.C==-8&&n.Vd(t,-9);},n.Fd=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O;if(R=new n.je(),e=new n.je(),D=new n.je(),k=new ht.bO(),g=new ht.bO(),P=n.Zl(t.L,0,A),i=n.Zl(t.L,1,A),t.zb(A,n.Ai(t.R,n.Ql(t.L,P)),n.Bi(t.R,n.Rl(t.L,P)),n.Ai(t.R,n.Ql(t.L,i)),n.Bi(t.R,n.Rl(t.L,i))),!n.Pm(t.L,P)&&!n.Pm(t.L,i)&&Rt(z(Pt(n.Ol(t.L,P),n.Ol(t.L,i)),i0),0)&&n.Vd(t,-8),t.o[P]?(R.a=t.o[P].a,R.c=t.o[P].b):(R.a=n.Ai(t.R,n.Ql(t.L,P)),R.c=n.Bi(t.R,n.Rl(t.L,P))),t.o[i]?(R.b=t.o[i].a,R.d=t.o[i].b):(R.b=n.Ai(t.R,n.Ql(t.L,i)),R.d=n.Bi(t.R,n.Rl(t.L,i))),(n.hm(t.L,A)&LA)!=0){n.Rd(t,R)&&t.rb(R),n.Vd(t,-9);return;}switch(u=n.im(t.L,A)==8?0:n.im(t.L,A)==16?1:n.fm(t.L,A),u){case 1:if(o=n.im(t.L,A),(t.G&128)!=0&&(o==257||o==129)&&(S=n.Zl(t.L,0,A),c=n.Hl(t.L,S),c!=0)){for(T=n.Gl(t.L,S),y=0,r=0;r<t.L.f;r++)n.Hl(t.L,r)==c&&n.Gl(t.L,r)==T&&++y;y==1&&(o=1);}switch(o){case 1:n.Rd(t,R)&&n.kd(t,R,P,i);break;case 257:n.Od(t,R,P,i);break;case 129:for(E=R.b-R.a,O=R.d-R.c,n.Dm(t.L,n.Xo(t.L,P,i))?(l=-3,U=-3):(l=t.p[P],U=n.qd(t,P),l==n.Cl(t.L,P)&&(l=U)),b=2;b<17;b+=2)e.a=R.a+b*E/17-b*O/128,e.c=R.c+b*O/17+b*E/128,e.b=R.a+b*E/17+b*O/128,e.d=R.c+b*O/17-b*E/128,n.Rd(t,e)&&(n.Vd(t,b<9?l:U),t.qb(e),n.Vd(t,t.Q));break;case 16:n.Rd(t,R)&&n.md(t,R,P,i);}break;case 0:case 2:if((t.s[P]||n.Qo(t.L,P)==2)&&(t.s[i]||n.Qo(t.L,i)==2)&&!n.gq(t.L,A)&&u==2){if(!n.Rd(t,R))break;n.Ad(t,R.b-R.a,R.d-R.c,k),E=k.a/2,O=k.b/2,e.a=R.a+E,e.c=R.c+O,e.b=R.b+E,e.d=R.d+O,D.a=R.a-E,D.c=R.c-O,D.b=R.b-E,D.d=R.d-O,n.im(t.L,A)==386&&n.Qd(e,D),n.kd(t,e,P,i),u==2?n.kd(t,D,P,i):n.hd(t,D,P,i);}else (t.s[i]||n.Qo(t.L,i)==2)&&u==2?n.Bd(t,R,A,false):(t.s[P]||n.Qo(t.L,P)==2)&&u==2?n.Bd(t,R,A,true):(F=n.Dp(t.L,A),F==0&&(F=1),e.a=R.a,e.c=R.c,e.b=R.b,e.d=R.d,n.Ad(t,R.b-R.a,R.d-R.c,k),F>0?(D.a=R.a+k.a,D.c=R.c+k.b,D.b=R.b+k.a,D.d=R.d+k.b,(n.zd(t,P,i,1,g)||n.$o(t.L,P)>1)&&(D.a+=g.a+k.b,D.c+=g.b-k.a),(n.zd(t,i,P,-1,g)||n.$o(t.L,i)>1)&&(D.b+=g.a-k.b,D.d+=g.b+k.a)):(D.a=R.a-k.a,D.c=R.c-k.b,D.b=R.b-k.a,D.d=R.d-k.b,(n.zd(t,P,i,-1,g)||n.$o(t.L,P)>1)&&(D.a+=g.a+k.b,D.c+=g.b-k.a),(n.zd(t,i,P,1,g)||n.$o(t.L,i)>1)&&(D.b+=g.a-k.b,D.d+=g.b+k.a)),n.im(t.L,A)==386&&n.Qd(e,D),n.Rd(t,e)&&n.kd(t,e,P,i),u==2?n.Rd(t,D)&&n.kd(t,D,P,i):n.Rd(t,D)&&n.hd(t,D,P,i));break;case 3:n.Rd(t,R)&&(n.kd(t,R,P,i),n.Ad(t,R.b-R.a,R.d-R.c,k),n.ld(t,R,P,i,k.a,k.b,e),n.ld(t,R,P,i,-k.a,-k.b,e));break;case 4:n.Rd(t,R)&&(n.Ad(t,R.b-R.a,R.d-R.c,k),n.ld(t,R,P,i,1.5*k.a,1.5*k.b,e),n.ld(t,R,P,i,.5*k.a,.5*k.b,e),n.ld(t,R,P,i,-0.5*k.a,-0.5*k.b,e),n.ld(t,R,P,i,-1.5*k.a,-1.5*k.b,e));break;case 5:n.Rd(t,R)&&(n.kd(t,R,P,i),n.Ad(t,R.b-R.a,R.d-R.c,k),n.ld(t,R,P,i,2*k.a,2*k.b,e),n.ld(t,R,P,i,k.a,k.b,e),n.ld(t,R,P,i,-k.a,-k.b,e),n.ld(t,R,P,i,-2*k.a,-2*k.b,e));}t.C==-8&&n.Vd(t,-9);},n.Gd=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g;for(c=false,r=0;r<t.L.g;r++)l=null,n.Cm(t.L,r)?(y=n._l(t.L,r),U=n.$l(t.L,r),l=y==U?"["+y+"]":"["+y+":"+U+"]"):(n.hm(t.L,r)&oe)!=0?l=(n.hm(t.L,r)&oe)==j0?"a":(n.hm(t.L,r)&384)==256?"r!a":"!a":(n.hm(t.L,r)&384)!=0&&(l=(n.hm(t.L,r)&384)==256?"r":"!r"),T=(n.hm(t.L,r)&iA)>>17,T!=0&&(l=(l??"")+T),l!=null&&(A=n.Zl(t.L,0,r),e=n.Zl(t.L,1,r),c||(t.Cb((t.X*2+1)/3|0),c=true),b=(n.Ai(t.R,n.Ql(t.L,A))+n.Ai(t.R,n.Ql(t.L,e)))/2,g=(n.Bi(t.R,n.Rl(t.L,A))+n.Bi(t.R,n.Rl(t.L,e)))/2,i=n.Ai(t.R,n.Ql(t.L,e))-n.Ai(t.R,n.Ql(t.L,A)),D=n.Bi(t.R,n.Rl(t.L,e))-n.Bi(t.R,n.Rl(t.L,A)),P=d.Math.sqrt(i*i+D*D),o=.6*t.vb(l),u=.55*t.wb(),P!=0&&(i>0?n.Id(t,b+o*D/P,g-u*i/P,l,true):n.Id(t,b-o*D/P,g+u*i/P,l,true)));c&&t.Cb(t.X);},n.Hd=function(t,A,e,r){s.sj(t.$,new ht.qO(A-t.V,e-t.V,2*t.V,2*t.V)),t.K||s.sj(t.U,new n.ie(A,e,n.wd(t,r)?-3:t.p[r]));},n.Id=function(t,A,e,r,P){var i,D,u;P&&(i=t.vb(r),D=i/2+(t.wb()/8|0),u=t.wb()/2|0,(r=="+"||r=="-")&&(u=u*2/3),s.sj(t.$,new ht.qO(A-D,e-u,2*D,2*u))),t.K||t.tb(r,A,e);},n.Jd=function(t){var A;A=t.a,t.a=t.b,t.b=A,A=t.c,t.c=t.d,t.d=A;},n.Kd=function(t,A,e){var r;return r=A==0?N1+t[0]-t[t.length-1]:t[A]-t[A-1],e>PP&&e<c0?r-=2*d.Math.cos(e+Sn):r-=.5*d.Math.cos(e+Sn),r;},n.Ld=function(t){var A;return A=new ht.pO(),t.a<=t.b?(A.c=t.a,A.b=t.b-t.a):(A.c=t.b,A.b=t.a-t.b),t.c<=t.d?(A.d=t.c,A.a=t.d-t.c):(A.d=t.d,A.a=t.c-t.d),A;},n.Md=function(t,A){var e,r,P,i,D,u,o;for(e=B(Q.Q7,jt,6,n.Lo(t.L,A),15,1),P=0;P<n.Lo(t.L,A);P++)e[P]=n.Yl(t.L,A,n.Zo(t.L,A,P));for(w.nAb(e,w.UAb(T0(s.Zrb.prototype.ld,s.Zrb,[]))),i=n.Nd(e,0),D=n.Kd(e,0,i),r=1;r<e.length;r++)u=n.Nd(e,r),o=n.Kd(e,r,u),D<o&&(D=o,i=u);return i;},n.Nd=function(t,A){var e;return A>0?(t[A]+t[A-1])/2:(e=t1+(t[0]+t[t.length-1])/2,e>t1?e-N1:e);},n.Od=function(t,A,e,r){var P,i,D,u;if(u=new n.je(),!(A.a==A.b&&A.c==A.d)){for(u.a=A.a,u.c=A.c,u.b=A.b,u.d=A.d,D=n.Ld(u),P=0;P<t.$.a.length;P++)if(i=s.xj(t.$,P),!(i.c>D.c+D.b||i.d>D.d+D.a||D.c>i.c+i.b||D.d>i.d+i.a)){if(n.Pd(t,u.a,u.c,P)){if(n.Pd(t,u.b,u.d,P))return;n.Sd(t,u,0,P),n.Od(t,u,e,r);return;}if(n.Pd(t,u.b,u.d,P)){n.Sd(t,u,1,P),n.Od(t,u,e,r);return;}}n.nd(t,u,e,r);}},n.Pd=function(t,A,e,r){var P;return (t.G&1)!=0?false:(P=s.xj(t.$,r),A>P.c&&A<P.c+P.b&&e>P.d&&e<P.d+P.a);},n.Qd=function(t,A){var e;e=t.b,t.b=A.b,A.b=e,e=t.d,t.d=A.d,A.d=e;},n.Rd=function(t,A){var e,r,P,i,D,u;if(A.a==A.b&&A.c==A.d){for(D=new s.trb(t.$);D.a<D.c.a.length;)if(i=s.srb(D),ht.lO(i,A.a,A.c))return  false;return  true;}for(u=n.Ld(A),e=false,A.a>A.b&&(n.Jd(A),e=true),r=0;r<t.$.a.length;r++)if(i=s.xj(t.$,r),!(i.c>u.c+u.b||i.d>u.d+u.a||u.c>i.c+i.b||u.d>i.d+i.a)){if(n.Pd(t,A.a,A.c,r))return n.Pd(t,A.b,A.d,r)?(e&&n.Jd(A),false):(n.Sd(t,A,0,r),P=n.Rd(t,A),e&&n.Jd(A),P);if(n.Pd(t,A.b,A.d,r))return n.Sd(t,A,1,r),P=n.Rd(t,A),e&&n.Jd(A),P;}return e&&n.Jd(A),true;},n.Sd=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b;e==0?(y=A.a,c=A.c,T=A.b,b=A.d):(y=A.b,c=A.d,T=A.a,b=A.c),U=s.xj(t.$,r),o=T>y?U.c+U.b:U.c,l=b>c?U.d+U.a:U.d,P=T-y,i=b-c,d.Math.abs(P)>d.Math.abs(i)?c==b?(D=o,u=c):(D=y+P*(l-c)/i,T>y==o>D?u=l:(D=o,u=c+i*(o-y)/P)):y==T?(D=y,u=l):(u=c+i*(o-y)/P,b>c==l>u?D=o:(D=y+P*(l-c)/i,u=l)),e==0?(A.a=D,A.c=u):(A.b=D,A.d=u);},n.Td=function(t,A){var e,r,P,i,D,u;if(t.L.q!=0){for(n.bw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),t.B=A,n.ed(t),P=n.gw(t.L),i=false,t.p=B(Q.S7,H,6,t.L.q,15,1),e=0;e<t.L.q;e++)t.p[e]=n.Cl(t.L,e),t.p[e]!=0&&(i=true),n.Pm(t.L,e)&&(t.p[e]=128),n.rm(t.L,e)&&(t.G&B1)==0&&(t.p[e]=256);for(n.Vd(t,-10),n.td(t),n.sd(t),n.ud(t),n.bd(t),t.Cb(t.X),t.Ab(t.Y),n.Vd(t,t.Q),n.xd(t),w.mAb(t.U.a,0),w.mAb(t.$.a,0),(t.G&1)!=0&&(n.Cd(t,P),n.Dd(t),n.Gd(t)),D=0;D<t.L.q;D++)n.wd(t,D)?(n.Vd(t,-3),n.Ed(t,D,P),n.Vd(t,t.Q)):t.p[D]!=0?(n.Vd(t,t.p[D]),n.Ed(t,D,P),n.Vd(t,t.Q)):!i&&t.L.O!=1&&n.Tl(t.L,D)!=1&&n.Tl(t.L,D)!=6&&(t.G&f0)==0&&n.Jl(t.L,D)==null&&n.Tl(t.L,D)<n.Xc.length&&n.Xc[n.Tl(t.L,D)]!=0?(u=n.Xc[n.Tl(t.L,D)],n.Yd(t,(r=t.M!=0?t.M:t.D!=0?t.D:-1,Ut.A4(u,r))),n.Ed(t,D,P),n.Vd(t,t.Q)):n.Ed(t,D,P);(t.G&1)==0&&(n.Dd(t),n.Gd(t),n.Cd(t,P));}},n.Ud=function(t,A,e,r){var P;switch(P=e/2,r&l1){case l1:if(A){t.w.a=A.c+A.b/2,t.w.b=A.d+A.a-P;break;}case 0:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d+t.v.a+P,A&&t.w.b>A.d+A.a-P&&(t.w.b=A.d+A.a-P);break;case sA:if(A){t.w.a=A.c+A.b/2,t.w.b=A.d+P;break;}case yA:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d-P,A&&t.w.b<A.d+P&&(t.w.b=A.d+P);}},n.Vd=function(t,A){if(!t.K&&(A==-10&&(t.C=-999,A=t.Q),A!=-2&&A!=-7&&t.N!=0&&(A=-4),A!=t.C&&!(t.C==-8&&A!=-9)))switch(A==-8&&(t.O=t.C),A==-9&&(A=t.O),t.C=A,A){case 0:t.Bb(t.F==0?G0:t.F);break;case  -6:t.Bb(t.F);break;case  -4:t.Bb(t.N);break;case  -2:t.Bb(t.t);break;case  -3:t.Bb(t.u);break;case  -7:t.Bb(t.H);break;case  -8:t.Bb(t.I);break;case  -5:t.Bb(t.P);break;case 64:t.Bb(-14655233);break;case 128:t.Bb(fA);break;case 256:t.Bb(-4194049);break;case 192:t.Bb(-16711936);break;case 320:t.Bb(-24576);break;case 384:t.Bb(-16744448);break;case 448:t.Bb(-6291456);break;case 1:t.Bb(qe);break;default:t.Bb(G0);}},n.Wd=function(t,A){t.G=A;},n.Xd=function(t,A,e){t.Q=-6,t.F=A,t.D=e,n._d(t);},n.Yd=function(t,A){if(t.N!=0){t.C!=-4&&(t.C=-4,t.Bb(t.N));return;}t.C=-5,t.P=A,t.Bb(A);},n.Zd=function(t){var A,e,r,P,i;for(P=n.Ai(t.R,n.Ql(t.L,0)),e=n.Ai(t.R,n.Ql(t.L,0)),i=n.Bi(t.R,n.Rl(t.L,0)),r=n.Bi(t.R,n.Rl(t.L,0)),A=0;A<t.L.q;A++)P>n.Ai(t.R,n.Ql(t.L,A))&&(P=n.Ai(t.R,n.Ql(t.L,A))),e<n.Ai(t.R,n.Ql(t.L,A))&&(e=n.Ai(t.R,n.Ql(t.L,A))),i>n.Bi(t.R,n.Rl(t.L,A))&&(i=n.Bi(t.R,n.Rl(t.L,A))),r<n.Bi(t.R,n.Rl(t.L,A))&&(r=n.Bi(t.R,n.Rl(t.L,A)));return new ht.qO(P,i,e-P,r-i);},n.$d=function(t,A,e){var r,P;return t.L.q==0?null:(t.v=n.Zd(t),r=t.R.c*n.Ul(t.L),P=new n.Ei(t.v,A,r,e),P.c==1&&P.a==0&&P.b==0?P=null:(n.ui(P,t.R),n.xi(P,t.v)),n.Ud(t,A,r,e),P);},n._d=function(t){var A;A=t.M!=0?t.M:t.D!=0?t.D:-1,t.t=Ut.B4(A),t.u=Ut.A4(Je,A),t.H=-24321,t.I=-6291392;},n.ae=function(t,A,e,r){var P;return n.be(t,A,e,r),n.zi(t.R)?null:(P=t.R,n.vi(t.R,t.L),t.R=new n.Ci(),P);},n.be=function(t,A,e,r){var P,i,D,u;if(t.L.q==0)return null;for(D=t.Db(e,r),n.bw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),n.xd(t),w.mAb(t.U.a,0),w.mAb(t.$.a,0),t.B=A,n.ed(t),t.Cb(t.X),t.K=true,i=0;i<t.L.q;i++)n.Ed(t,i,null);return t.K=false,P=t.R.c*n.Ul(t.L),n.od(t,P),n.Ud(t,e,P,r),!e||ht.mO(e,t.v)?D:(u=new n.Ei(t.v,e,P,r),n.ui(u,t.R),n.xi(u,t.v),n.wi(u,t.w),D?(n.ui(u,D),D):u);},n.ce=function(t){n.de.call(this,t,0);},n.de=function(t,A){n.ad(this),this.L=t,this.G=A,this.xb();},I(121,1,{121:1}),L.xb=function(){n.vd(this);},L.yb=function(t,A,e,r){},L.zb=function(t,A,e,r,P){},L.Db=function(t,A){return n.$d(this,t,A);},L.r=0,L.t=0,L.u=0,L.A=0,L.C=0,L.D=0,L.F=0,L.G=0,L.H=0,L.I=0,L.J=0,L.K=false,L.M=0,L.N=0,L.O=0,L.P=0,L.Q=0,L.S=0,L.T=0,L.V=0,L.W=0,L.X=0,L.Y=0,L.Z=0,n.Yc=.15,n.Zc=.06,n.$c=.6,Q.X7=N(121),n.ie=function(t,A,e){this.b=t,this.c=A,this.a=e;},I(144,1,{144:1},n.ie),L.a=0,L.b=0,L.c=0,Q.V7=N(144),n.je=function(){},I(68,1,{},n.je),L.a=0,L.b=0,L.c=0,L.d=0,Q.W7=N(68),n.ke=function(t,A){var e;if(t.i==null)return  false;for(e=0;e<t.i.length;e++)if(!A.Hb(t.i[e].a,t.i[e].b))return  false;return  true;},n.le=function(t,A,e){var r;if(t.i!=null)for(r=0;r<t.i.length;r++)t.i[r].a+=A,t.i[r].b+=e;},n.me=function(t,A){var e;if(t.i!=null)for(e=0;e<t.i.length;e++)t.i[e].a*=A,t.i[e].b*=A;},n.ne=function(t,A){t.g=A;},n.oe=function(t,A,e){var r;if(t.i!=null)for(r=0;r<t.i.length;r++)t.i[r].a=t.o[r]+A-t.k,t.i[r].b=t.p[r]+e-t.n;},n.pe=function(t,A,e){var r,P;if(t.k=A,t.n=e,t.i!=null)for(P=t.i.length,t.o=B(Q.Q7,jt,6,P,15,1),t.p=B(Q.Q7,jt,6,P,15,1),r=0;r<P;r++)t.o[r]=t.i[r].a,t.p[r]=t.i[r].b;},n.qe=function(t,A,e){var r,P,i;if(t.i!=null)for(r=0;r<t.i.length;r++)i=t.p[r]*A,P=t.o[r]-e,t.i[r].a=t.k+i*d.Math.sin(P),t.i[r].b=t.n+i*d.Math.cos(P);},n.re=function(t,A,e){var r,P,i,D;if(t.k=A,t.n=e,t.i!=null)for(D=t.i.length,t.o=B(Q.Q7,jt,6,D,15,1),t.p=B(Q.Q7,jt,6,D,15,1),i=0;i<D;i++)r=A-t.i[i].a,P=e-t.i[i].b,t.p[i]=d.Math.sqrt(r*r+P*P),t.o[i]=n.no(A,e,t.i[i].a,t.i[i].b);},I(193,1,{193:1}),L.Jb=function(t){var A,e;if(this.i==null)return null;for(A=new ht.qO(this.i[0].a,this.i[0].b,0,0),e=1;e<this.i.length;e++)A.c>this.i[e].a?(A.b+=A.c-this.i[e].a,A.c=this.i[e].a):A.b<this.i[e].a-A.c&&(A.b=this.i[e].a-A.c),A.d>this.i[e].b?(A.a+=A.d-this.i[e].b,A.d=this.i[e].b):A.a<this.i[e].b-A.d&&(A.a=this.i[e].b-A.d);return A;},L.Nb=function(t){n.me(this,t);},L.pb=function(){var t;return t=new h.Ekb(),h.Dkb(t,iP+this.Lb()+'"'+this.Kb()+Ze),t.a;},L.Ob=function(t,A){n.oe(this,t,A);},L.Pb=function(t,A){n.qe(this,t,A);},L.Qb=function(t,A){n.re(this,t,A);},L.g=false,L.j=false,L.k=0,L.n=0,Q.Y7=N(193),n.ye=function(t,A,e){if(t.f[A])return 0;if(n.Tl(t.j,A)==7){if(n.Lo(t.j,A)==3)return 6;if(n.$o(t.j,A)==2)return t.a?0:4;}else {if(n.Tl(t.j,A)==8)return 10;if(n.Tl(t.j,A)==15||n.Tl(t.j,A)==33){if(n.$o(t.j,A)==3)return 8;}else if(n.Tl(t.j,A)==16||n.Tl(t.j,A)==34||n.Tl(t.j,A)==52){if(n.$o(t.j,A)==2)return 11;if(n.$o(t.j,A)==3)return n.Bl(t.j,A)==1?12:(e&&n.jn(t.j,A,1),5);}else if(n.Tl(t.j,A)==6)return n.Bl(t.j,A)==-1?n.Lo(t.j,A)==3?16:n.Mo(t.j,A)==3?15:14:(e&&n.jn(t.j,A,-1),n.Lo(t.j,A)!=n.Mo(t.j,A)?2:3);}return 0;},n.ze=function(t,A,e){if(t.f[A])return 0;if(t.a){if(n.Lo(t.j,A)!=3)return 0;}else if(n.Lo(t.j,A)>3)return 0;return n.Tl(t.j,A)==6?(e&&n.jn(t.j,A,1),2):n.Tl(t.j,A)==5?4:0;},n.Ae=function(t,A,e){if(n.Bl(t.j,A)!=0)return 0;if(t.a){if(n.Tl(t.j,A)==5)return n.vp(t.j,A)!=2?0:(e&&n.jn(t.j,A,1),1);if(n.Tl(t.j,A)==7)return n.vp(t.j,A)!=2?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?6:3);if(n.Tl(t.j,A)==8)return n.vp(t.j,A)!=1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?7:4);if(n.Tl(t.j,A)==16)return n.vp(t.j,A)!=1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?5:2);if(n.Tl(t.j,A)==34)return n.vp(t.j,A)!=1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?4:1);}else {if(n.Tl(t.j,A)==5)return n.vp(t.j,A)>2?0:(e&&n.jn(t.j,A,1),1);if(n.Tl(t.j,A)==7)return n.vp(t.j,A)>2?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?5:3);if(n.Tl(t.j,A)==8)return n.vp(t.j,A)>1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?7:4);if(n.Tl(t.j,A)==16)return n.vp(t.j,A)>1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?5:2);}return 0;},n.Be=function(t,A,e){var r,P,i;if(r=n.Tl(t.j,A),r>=5&&r<=8||r==15||r==16||r==33||r==34||r==52){if(i=n.pp(t.j,A),i!=0)return  true;if(P=n.Bl(t.j,A),r==5&&P>=0)return e&&n.jn(t.j,A,P-1),true;if(r!=5&&P<=0)return e&&n.jn(t.j,A,P+1),true;}return  false;},n.Ce=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b;for(A=0;A<t.j.f;A++)if(t.d[A])for(D=false,u=B(Q.S7,H,6,t.j.f,15,1),b=B(Q.S7,H,6,t.j.f,15,1),o=B(Q.S7,H,6,t.j.f,15,1),u[0]=A,b[A]=-1,o[A]=1,P=0,l=0;P<=l&&!D;){for(i=u[P],U=0;U<n.$o(t.j,i)&&!D;U++)if(y=(o[i]&1)==1^n.fm(t.j,n._o(t.j,i,U))>1,r=n.Zo(t.j,i,U),o[r]==0&&y)if(t.d[r]){if((o[i]&1)==1){for(t.d[A]=false,t.d[r]=false,t.k+=2,c=i,T=0;T<o[i];T++)e=n.Xo(t.j,r,c),t.e[e]&&(t.e[e]=false,--t.c),n.In(t.j,e,n.fm(t.j,e)==1?2:n.fm(t.j,e)-1),r=c,c=b[r];D=true;}}else u[++l]=r,b[r]=i,o[r]=o[i]+1;++P;}},n.De=function(t,A){var e;for(e=0;e<n.$o(t.j,A);e++)if(n.Lm(t.j,n.Zo(t.j,A,e)))return  true;return  false;},n.Ee=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F;if(A!=null)t.e=A;else for(t.e=B(Q.xfb,Tt,6,t.j.g,16,1),T=0;T<t.j.g;T++)n.im(t.j,T)==8&&(t.e[T]=true,n.Mn(t.j,T,1));for(c=0;c<t.j.g;c++)t.e[c]&&++t.c;for(t.d=B(Q.xfb,Tt,6,t.j.f,16,1),t.f=B(Q.xfb,Tt,6,t.j.f,16,1),u=0;u<t.j.f;u++){for(b=0,g=0;g<n.$o(t.j,u);g++)t.e[n._o(t.j,u,g)]&&++b;b>0&&(t.d[u]=true,++t.b,b==3&&(t.f[u]=true));}for(t.i=B(Q.xfb,Tt,6,e.j.a.length,16,1),F=0;F<e.j.a.length;F++)for(t.i[F]=true,l=s.xj(e.k,F),U=0,y=l.length;U<y;++U)if(o=l[U],!t.e[o]){t.i[F]=false;break;}for(t.g=B(Q.xfb,Tt,6,t.j.f,16,1),k=0;k<e.j.a.length;k++)if(t.i[k]&&s.xj(e.k,k).length!=6)for(P=s.xj(e.j,k),i=0,D=P.length;i<D;++i)r=P[i],s.xj(e.k,k).length==5&&(t.g[r]=true);t.k=0;},n.Fe=function(t,A,e){var r,P;for(P=0;P<n.$o(t.j,A);P++)if(r=n.Zo(t.j,A,P),n.Tl(t.j,r)==8&&n.$o(t.j,r)==1)return e&&n.Bl(t.j,r)==0&&n.jn(t.j,r,-1),true;return  false;},n.Ge=function(t,A,e){var r,P,i,D;if(n.bw(t.j,1),D=new n.Dt(t.j,1),n.Ee(t,A,D),t.c==0)return  true;if(e)for(P=0;P<t.j.f;P++)t.d[P]&&n.Tl(t.j,P)==7&&(t.f[P]||n.$o(t.j,P)==3&&!t.g[P])&&n.jn(t.j,P,1);for(t.a=false,n.Re(t,e),t.j.K&&n.Je(t),n.Te(t,D),n.Oe(t,D),n.Qe(t),n.Ke(t);n.Me(t,D,e);)n.Ke(t);for(;n.Ne(t,D,e);)n.Ke(t);for(;n.Le(t,D,e);)n.Ke(t);for(;t.c!=0;)for(i=0;i<t.j.g;i++)t.e[i]&&(n.Ie(t,i),n.Ke(t));for(e&&n.Ue(t,e),t.b-t.k>=2&&n.Ce(t),r=0;r<t.j.f;r++)t.d[r]&&n.op(t.j,r)!=0&&(e&&(n.Bl(t.j,r)==1&&n.Hm(t.j,r)||n.Bl(t.j,r)==-1&&n.Tl(t.j,r)==5)?n.jn(t.j,r,0):n.wn(t.j,r,32),++t.k);return t.b==t.k;},n.He=function(t,A){return t==A-1?0:t+1;},n.Ie=function(t,A){var e,r,P,i;for(n.im(t.j,A)==1&&(n.Mn(t.j,A,2),t.k+=2),P=0;P<2;P++)for(e=n.Zl(t.j,P,A),t.d[e]=false,i=0;i<n.$o(t.j,e);i++)r=n._o(t.j,e,i),t.e[r]&&(t.e[r]=false,--t.c);},n.Je=function(t){var A,e,r,P,i,D,u,o;for(e=0;e<t.j.g;e++)if(t.e[e])for(P=0;P<2;P++){for(u=n.Zl(t.j,P,e),A=false,D=0;D<n.$o(t.j,u);D++)if(e!=n._o(t.j,u,D)&&t.e[n._o(t.j,u,D)]){A=true;break;}if(!A){for(o=e,r=n.Zl(t.j,1-P,e);o!=-1;)for(t.d[u]=false,t.e[o]=false,--t.c,n.Mn(t.j,o,8),o=-1,u=r,i=0;i<n.$o(t.j,u);i++)if(t.e[n._o(t.j,u,i)])if(o==-1)o=n._o(t.j,u,i),r=n.Zo(t.j,u,i);else {u=-1,o=-1;break;}u!=-1&&(t.d[r]=false);break;}}},n.Ke=function(t){var A,e,r,P,i,D,u;do for(u=false,e=0;e<t.j.g;e++)if(t.e[e]){for(i=false,P=0;P<2;P++){for(r=n.Zl(t.j,P,e),A=false,D=0;D<n.$o(t.j,r);D++)if(e!=n._o(t.j,r,D)&&t.e[n._o(t.j,r,D)]){A=true;break;}if(!A){i=true;break;}}i&&(u=true,n.Ie(t,e));}while(u);},n.Le=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k;for(b=0;b<A.j.a.length;b++)if(s.xj(A.k,b).length!=6&&t.i[b]){for(y=true,k=s.xj(A.k,b),U=0;U<k.length;U++)if(!t.e[k[U]]){y=false;break;}if(y){for(u=-1,o=0,g=s.xj(A.j,b),l=0;l<g.length;l++)r=g[l],c=t.g[r]?n.ye(t,r,false):n.ze(t,r,false),o<c&&(o=c,u=l);if(u!=-1){if(T=g[u],e){for(P=g,i=0,D=P.length;i<D;++i)r=P[i],r!=T&&n.Be(t,r,true);t.g[T]?n.ye(t,T,true):n.ze(t,T,true);}return n.Pe(t,T),t.g[T]&&n.Se(t,g,k,u),true;}}}return  false;},n.Me=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S;for(S=B(Q.S7,H,6,t.j.g,15,1),g=0;g<A.j.a.length;g++){for(F=s.xj(A.k,g),y=true,o=0;o<F.length;o++)if(!t.e[F[o]]){y=false;break;}if(y)for(u=0;u<F.length;u++)++S[F[u]];}for(P=t.c,b=0;b<A.j.a.length;b++)for(D=false,k=s.xj(A.j,b),F=s.xj(A.k,b),u=0;u<F.length&&!D;u++)if(S[F[u]]>1&&(i=n.He(u,F.length),S[F[i]]==1)){for(l=false,c=n.He(i,F.length);S[F[c]]==1;)(!n.Be(t,k[c],false)||F.length!=6&&n.Tl(t.j,k[c])!=6)&&(l=true),c=n.He(c,F.length);if(!l&&(r=c>i?c-i:c+F.length-i,r>2&&(r&1)==1)){for(T=1;T<r;T+=2)U=i+T<F.length?i+T:i+T-F.length,e&&(n.Be(t,k[U],true),n.Be(t,k[U==k.length-1?0:U+1],true)),n.Ie(t,F[U]);D=true;}}return P!=t.c;},n.Ne=function(t,A,e){var r,P,i,D,u,o,l,U;for(r=false,o=0;o<A.j.a.length;o++)if(s.xj(A.k,o).length==6){for(u=true,l=s.xj(A.j,o),U=s.xj(A.k,o),i=0;i<6;i++)if(!n.Be(t,l[i],false)||!t.e[U[i]]){u=false;break;}if(u){if(e)for(D=0;D<6;D++)n.Be(t,l[D],true);for(P=0;P<6;P+=2)n.Ie(t,U[P]);r=true;break;}}return r;},n.Oe=function(t,A){var e;for(e=0;e<t.j.g;e++)t.e[e]&&n.wt(A,e)&&(n.Pe(t,n.Zl(t.j,0,e)),n.Pe(t,n.Zl(t.j,1,e)));},n.Pe=function(t,A){var e,r;for(t.d[A]&&(t.d[A]=false,--t.b),r=0;r<n.$o(t.j,A);r++)e=n._o(t.j,A,r),t.e[e]&&(t.e[e]=false,--t.c);},n.Qe=function(t){var A,e,r,P,i;for(e=0;e<t.j.g;e++)if(n.fm(t.j,e)==2){for(P=0;P<2;P++)if(A=n.Zl(t.j,P,e),n.Tl(t.j,A)<=8){for(i=0;i<n.$o(t.j,A);i++)if(r=n._o(t.j,A,i),t.e[r]){n.Pe(t,A);break;}}}},n.Re=function(t,A){var e;for(e=0;e<t.j.f;e++)t.d[e]&&n.pp(t.j,e)==0&&(!A||n.Tl(t.j,e)==5&&n.Bl(t.j,e)<0||n.Tl(t.j,e)==6||n.Tl(t.j,e)==14||n.Hm(t.j,e)&&n.Bl(t.j,e)>0)&&n.Pe(t,e);},n.Se=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k;if(y=A[r==0?4:r-1],k=A[r==4?0:r+1],n.$o(t.j,y)==3&&n.$o(t.j,k)==3){for(c=B(Q.S7,H,6,t.j.f,15,1),T=B(Q.xfb,Tt,6,t.j.g,16,1),i=e,D=0,u=i.length;D<u;++D)o=i[D],T[o]=true;for(P=0;P<t.j.g;P++)t.e[P]||(T[P]=true);if(b=n.yp(t.j,c,y,k,14,null,T),b==14){for(g=true,U=2;U<=12;U++)if((U&3)!=1^n.Uo(t.j,c[U])==5){g=false;break;}if(g)for(n.Pe(t,c[7]),l=1;l<=5;l+=2)n.Ie(t,n.Xo(t.j,c[l],c[l+1])),n.Ie(t,n.Xo(t.j,c[l+7],c[l+8]));}}},n.Te=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K;for(R=B(Q.xfb,Tt,6,t.j.f,16,1),V=0;V<A.j.a.length;V++)if(At=s.xj(A.k,V).length,(At==3||At==5||At==6||At==7)&&t.i[V])for(r=s.xj(A.j,V),P=0,i=r.length;P<i;++P)e=r[P],R[e]=true;for(T=B(Q.S7,H,6,t.j.f,15,1),b=B(Q.xfb,Tt,6,t.j.f,16,1),l=0;l<t.j.g;l++)D=n.Zl(t.j,0,l),u=n.Zl(t.j,1,l),!R[D]&&!R[u]&&(t.e[l]&&(++T[D],++T[u]),n.im(t.j,l)==16&&(b[D]=true,b[u]=true));for(J=B(Q.S7,H,6,t.j.f,15,1),c=B(Q.S7,H,6,t.j.f,15,1),K=0;K<t.j.f;K++)if(T[K]==1)for(c[0]=K,y=0,g=0;y<=g;){for(k=0;k<n.$o(t.j,c[y]);k++)if(t.e[n._o(t.j,c[y],k)]&&(U=n.Zo(t.j,c[y],k),(y==0||U!=c[y-1])&&T[U]!=0)){if(c[++g]=U,(T[U]&1)!=0){for(O=1;O<g;O+=2)J[c[O]]=-1;g=0;}break;}++y;}for(o=B(Q.xfb,Tt,6,t.j.f,16,1),ot=0;ot<t.j.f;ot++)if(!o[ot]&&T[ot]!=0){for(c[0]=ot,o[ot]=true,y=0,g=0;y<=g;){for(k=0;k<n.$o(t.j,c[y]);k++)t.e[n._o(t.j,c[y],k)]&&(U=n.Zo(t.j,c[y],k),o[U]||(c[++g]=U,o[U]=true));++y;}if((g&1)==0){for(F=0;F<=g;F++)J[c[F]]==0&&(J[c[F]]=n.Ae(t,c[F],false));for(E=true,S=0;S<=g;S++)if(J[c[S]]<=0&&!n.Be(t,c[S],false)){E=false;break;}if(E){for(G=0,x=-1,k=0;k<=g;k++)G<J[c[k]]&&(G=J[c[k]],x=c[k]);G>0&&(n.Ae(t,x,true),n.Pe(t,x));}}}},n.Ue=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g;for(e=0;e<t.j.f;e++)if(n.bw(t.j,1),n.Tl(t.j,e)==7&&n.Bl(t.j,e)==1&&n.Qo(t.j,e)==1&&!n.Fe(t,e,A))for(u=false,o=B(Q.S7,H,6,t.j.f,15,1),g=B(Q.S7,H,6,t.j.f,15,1),l=B(Q.S7,H,6,t.j.f,15,1),o[0]=e,g[e]=-1,l[e]=1,i=0,U=0;i<=U&&!u;){for(D=o[i],y=0;y<n.$o(t.j,D)&&!u;y++)if(T=(l[D]&1)==0^n.fm(t.j,n._o(t.j,D,y))>1,P=n.Zo(t.j,D,y),l[P]==0&&T)if(n.Tl(t.j,P)==7&&n.Qo(t.j,P)==0&&n.$o(t.j,P)==2){if((l[D]&1)==0){for(n.jn(t.j,e,0),b=D,c=0;c<l[D];c++)r=n.Xo(t.j,P,b),n.In(t.j,r,n.fm(t.j,r)==1?2:n.fm(t.j,r)-1),P=b,b=g[P];u=true;}}else o[++U]=P,g[P]=D,l[P]=l[D]+1;++i;}},n.Ve=function(t){n.We.call(this,t);},n.We=function(t){this.j=t;},I(89,1,{},n.Ve),L.a=false,L.b=0,L.c=0,L.k=0,Q.Z7=N(89),n.Xe=function(t,A){var e,r;for(e=0,r=0;r<t.j[A];r++)t.n[A][r]==2&&(n.Tl(t,t.i[A][r])==7||n.Tl(t,t.i[A][r])==8||n.Tl(t,t.i[A][r])==16)&&++e;return e;},n.Ye=function(t,A){var e,r,P,i,D,u;if(t.s[A]==0)return  false;for(u=true,e=t.s[A],i=t.j[A],D=0,r=0;r<i;r++)P=t.i[A][r],D+=t.s[P];return d.Math.abs(e)<=d.Math.abs(D)&&h.Ljb(e)!=h.Ljb(D)&&(u=false),u;},n.Ze=function(t,A){var e,r,P,i,D,u,o;if(t.C[A]!=8||t.s[A]!=0||t.j[A]!=1||t.n[A][0]!=1)return  false;if(e=t.i[A][0],t.C[e]==6){for(D=t.j[e],r=0;r<D;r++)if(P=t.i[e][r],P!=A&&t.C[P]==8&&(i=n.Xo(t,e,P),t.H[i]==2))return  true;}else if(t.C[e]==7){if(t.s[e]==1)return  true;}else if(t.C[e]==16){for(u=t.j[e],o=0,r=0;r<u;r++)P=t.i[e][r],P!=A&&t.C[P]==8&&(i=n.Xo(t,e,P),t.H[i]==2&&++o);if(o==2)return  true;}else if(n.$e(t,A))return  true;return  false;},n.$e=function(t,A){var e,r,P,i,D;if(t.C[A]!=8||t.j[A]!=1)return  false;if(e=t.i[A][0],t.C[e]==15){for(D=t.j[e],r=0;r<D;r++)if(P=t.i[e][r],P!=A&&t.C[P]==8&&(i=n.Xo(t,e,P),t.H[i]==2))return  true;}return  false;},n._e=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G;if(t.C[A]!=7||t.s[A]!=0||t.j[A]+t.o[A]>3)return  false;if(A<t.f&&n.ot(t.p,A)){if(t.o[A]!=1||n.To(t,A,7)!=1)return  false;for(E=(n.bw(t,7),t.p),S=0;S<E.j.a.length;S++)if(n.qt(E,S,A)){if(s.xj(E.k,S).length==5||s.xj(E.k,S).length==6){for(O=s.xj(E.j,S),k=-1,o=0;o<O.length;o++)if(O[o]==A){k=o;break;}for(P=0,F=null,g=null,O.length==5&&(F=B(Q.S7,H,6,2,15,1),F[0]=O[k-1<0?k+4:k-1],F[1]=O[k-4<0?k+1:k-4],g=B(Q.S7,H,6,2,15,1),g[0]=O[k-2<0?k+3:k-2],g[1]=O[k-3<0?k+2:k-3]),O.length==6&&(F=B(Q.S7,H,6,3,15,1),F[0]=O[k-1<0?k+5:k-1],F[1]=O[k-3<0?k+3:k-3],F[2]=O[k-5<0?k+1:k-5],g=B(Q.S7,H,6,2,15,1),g[0]=O[k-2<0?k+4:k-2],g[1]=O[k-4<0?k+2:k-4]),l=0;l<O.length;l++)A!=O[l]&&n.Tl(t,O[l])==7&&n.Qo(t,O[l])==1&&--P;for(U=0;U<F.length;U++){for(i=-1,D=-1,b=0;b<n.$o(t,F[U]);b++)if(!n.Up(t,n._o(t,F[U],b))){i=n.Zo(t,F[U],b),D=n._o(t,F[U],b);break;}if(i!=-1){if(t.C[i]==7&&t.o[i]==0&&t.j[i]+t.o[i]<=3&&!n.af(t,i,false)){++P;continue;}if(t.C[i]==8&&t.j[i]==1){P+=2;continue;}if(D<t.g&&n.pt(t.p,D)){for(x=0;x<E.j.a.length;x++)if(E.e[x]&&n.qt(E,x,i)){for(R=s.xj(E.j,x),c=0;c<R.length;c++)if(n.Tl(t,R[c])==7&&n.Qo(t,R[c])==1){--P;break;}break;}}}}for(y=0;y<g.length;y++){for(i=-1,c=0;c<n.$o(t,g[y]);c++)n.Up(t,n._o(t,g[y],c))||(i=n.Zo(t,g[y],c));n.Tl(t,g[y])==6?i!=-1&&n.Xe(t,i)!=0&&--P:n.Tl(t,g[y])==7&&n.Qo(t,g[y])==0&&(i==-1||!(i<t.f&&n.ot(t.p,i))&&n.Xe(t,i)==0)&&++P;}return P>0;}break;}return  false;}if(t.o[A]>1)return  false;if(t.o[A]==1){for(T=-1,G=0,o=0;o<t.j[A];o++){if(r=t.i[A][o],t.n[A][o]==2){if(t.C[r]!=6)return  false;T=r;continue;}if(t.C[r]==8)return  false;if(t.C[r]==7){--G,n.af(t,r,false)&&--G;continue;}r<t.f&&n.ot(t.p,r)&&--G;}if(T==-1)return  false;for(e=0,l=0;l<t.j[T];l++)if(t.n[T][l]==1){if(r=t.i[T][l],n.Xe(t,r)!=0)return  false;r<t.f&&n.ot(t.p,r)&&++e,t.C[r]==7&&!n.af(t,r,true)&&++G,(t.C[r]==8||t.C[r]==16)&&--G;}return e==2&&--G,G>=0;}for(u=0;u<t.j[A];u++)if(r=t.i[A][u],r<t.f&&n.ot(t.p,r)||t.C[r]!=6||n.Xe(t,r)!=0||t.o[r]!=0&&n.bf(t,r))return  false;return  true;},n.af=function(t,A,e){var r,P,i;for(r=false,i=0;i<t.j[A];i++)if(!n.Up(t,t.k[A][i])&&t.n[A][i]==1&&(P=t.i[A][i],!(P<t.f&&n.ot(t.p,P))&&(t.C[P]==6&&n.Xe(t,P)==1||t.C[P]==16&&n.Xe(t,P)==2))){if(r||!e)return  true;r=true;}return  false;},n.bf=function(t,A){var e,r,P;for(r=0;r<t.j[A];r++)if(t.n[A][r]!=1){for(e=t.i[A][r],P=0;P<t.j[e];P++)if(t.n[e][P]==1&&n.Xe(t,t.i[e][P])!=0)return  true;}return  false;},n.ef=function(){n.ef=i1,n.cf=j(v(Q.wfb,1),Yt,6,15,[-1,-1,-1,0,0,1,2,3,4,5,-1,0,0,0,6,7,8,9,-1,0,0,10,10,10,10,10,10,10,10,10,10,1,11,11,12,13,-1,0,0,10,10,10,10,10,10,10,10,10,10,0,0,0,11,14,-1,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,10,10,10,10,10,10,10,10,1,1,1,1,-1,-1,-1,-1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]),n.df=j(v(Q.wfb,1),Yt,6,15,[-1,-1,-1,0,0,0,2,5,5,5,-1,0,0,0,0,9,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);},n.ff=function(t,A,e){n.ef();var r,P,i,D,u,o,l,U,y,T,c,b,g,k;for(n.bw(t,7),c=B(Q.T7,e0,6,t.j[A],14,1),T=0,o=0;o<t.j[A];o++)if(P=t.i[A][o],t.C[P]!=1){if(D=0,(e&32)!=0&&(i=Mt(t.n[A][o]),e==32190?nt(i,3)<0&&n.Yp(t,t.k[A][o])&&t.o[A]==1&&(i=0):nt(i,3)<0&&n.Up(t,t.k[A][o])&&(i=0),D=Pt(D,c1(i,4))),(e&128)!=0){if(n.cf[t.C[P]]==-1)throw Lt(new h.cC(fP+t.C[P]));D=v1(D,Mt(n.cf[t.C[P]]));}else if((e&64)!=0){if(n.df[t.C[P]]==-1)throw Lt(new h.cC(fP+t.C[P]));D=v1(D,Mt(n.df[t.C[P]]));}for((e&256)!=0&&(b=t.j[P]-1,b>3&&(b=3),(e&512)==0&&b>1&&(b=1),D=Pt(D,Q.Mfb(b<<6))),(e&f0)!=0&&(t.u[P]&8)!=0&&(D=Pt(D,256)),(e&B1)!=0&&P<t.f&&n.ot(t.p,P)&&(D=Pt(D,512)),(e&N0)!=0&&(t.u[P]&U0)!=0&&(D=Pt(D,f0)),U=0;nt(D,c[U])<0;)++U;for(y=o;y>U;y--)c[y]=c[y-1];c[U]=D,++T;}for(T>4&&(T=4),r=0,l=0;l<T;l++)r=c1(r,11),r=Pt(r,c[l]);if(r=c1(r,15),n.cf[t.C[A]]==-1)throw Lt(new h.cC(fP+t.C[A]));if(r=Pt(r,Mt(n.cf[t.C[A]])),(e&2)!=0?(k=t.p&&A<t.f?n.dt(t.p,A):0,k>9&&(k=9),k>2&&(k-=2),r=Pt(r,Q.Mfb(k<<4))):(e&1)!=0&&(t.u[A]&8)!=0&&(r=Pt(r,64)),(e&4)!=0&&A<t.f&&n.ot(t.p,A)&&(r=Pt(r,f0)),(e&8)!=0&&(t.u[A]&Z1)!=0&&(r=Pt(r,B1)),(e&16)!=0&&(t.u[A]&U0)!=0&&(r=Pt(r,Z1)),(e&Z1)!=0&&(n.Ye(t,A)&&(r=Pt(r,U0)),n._e(t,A))){for(u=0;u<t.f;u++)if(n.Ze(t,u)){r=Pt(r,N0);break;}}return (e&U0)!=0&&(g=Mt(n.To(t,A,10)),r=Pt(r,c1(g,7))),r;},n.gf=function(t,A,e,r,P,i){var D,u,o,l;for(o=0,u=0;u<t.R.f;u++)Rt(z(n.Ol(t.R,t.B[u]),r),0)&&++o;if(o!=0)for(n.eg(t,A),n._f(t,Q.Mfb(o),e),D=0;D<t.R.f;D++)l=z(n.Ol(t.R,t.B[D]),r),nt(l,0)!=0&&(n._f(t,Q.Mfb(D),e),P!=1&&n._f(t,A0(l,i),P));},n.hf=function(t,A,e,r,P,i){var D,u,o,l;for(o=0,u=0;u<t.R.g;u++)(n.hm(t.R,t.C[u])&r)!=0&&++o;if(o!=0)for(n.eg(t,A),n._f(t,Q.Mfb(o),e),D=0;D<t.R.g;D++)l=n.hm(t.R,t.C[D])&r,l!=0&&(n._f(t,Q.Mfb(D),e),P!=1&&n._f(t,Q.Mfb(l>>i),P));},n.jf=function(t){for(var A,e,r,P,i,D;;){for(i=B(Q.xfb,Tt,6,t.U+1,16,1),r=-1,e=0;e<t.R.f;e++)n.pp(t.R,e)!=0&&(i[t.f[e]]&&r<t.f[e]&&(r=t.f[e]),i[t.f[e]]=true);if(r==-1)break;for(P=0,A=0;A<t.R.f;A++)D=0,t.f[A]==r&&(D=++P),n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),n.ah(t.e[A],8,Q.Mfb(D));t.U=n.Kf(t);}},n.kf=function(t){var A,e,r,P;for(e=0;e<t.R.f;e++)n.eh(t.e[e],e),n.ah(t.e[e],2*t.c+4,c1(Mt(t.f[e]),t.c+4));for(P=false,A=0;A<t.R.f;A++)P=P|n.xf(t,A,3);for(r=0;r<t.R.g;r++)P=P|n.sf(t,r,3);if(!P)return  false;for(;t.U<t.R.f&&(P=n.Hf(t),!!P);)n.Jf(t),t.P&&n.uh(t.P,t.f);return  true;},n.lf=function(t){var A,e,r,P,i;for(e=0;e<t.R.f;e++)n.eh(t.e[e],e),n.ah(t.e[e],t.c+1,V0(2,Mt(t.f[e])));for(i=B(Q.S7,H,6,t.U+1,15,1),r=0;r<t.R.f;r++)++i[t.f[r]];for(P=1;i[P]==1;)++P;for(A=0;A<t.R.f;A++)if(t.f[A]==P){n.bh(t.e[A],1);break;}t.U=n.Kf(t);},n.mf=function(t,A,e){var r,P,i,D,u;return n.Tl(t.R,A)!=6&&n.Tl(t.R,A)!=7||(P=n.Zo(t.R,A,0),i=n.Zo(t.R,A,1),n.Qo(t.R,P)!=1||n.Qo(t.R,i)!=1)||n.$o(t.R,P)==1||n.$o(t.R,i)==1||n.Lo(t.R,P)>3||n.Lo(t.R,i)>3||(D=new n.lk(t.R,t.f,A,P),D.f&&e==1)||(u=new n.lk(t.R,t.f,A,i),u.f&&e==1)||D.f&&u.f?false:(e==3&&(D.f&&D.c&&(t.W[A]=true),u.f&&u.c&&(t.W[A]=true)),r=t.jb?n.of(t,D,u):n.nf(D,u),e==1?t.db[A]=r:e==2&&(D.f&&(r==1?n.bh(t.e[D.b],Mt(t.f[P])):n.bh(t.e[D.d],Mt(t.f[P]))),u.f&&(r==2?n.bh(t.e[u.b],Mt(t.f[i])):n.bh(t.e[u.d],Mt(t.f[i])))),true);},n.nf=function(t,A){var e,r,P;if(r=n.kk(t),P=n.kk(A),r==-1||P==-1||(r+P&1)==0)return 3;switch(e=0,r+P){case 3:case 7:e=2;break;case 5:e=1;}return e;},n.of=function(t,A,e){var r,P;return r=B(Q.S7,H,6,4,15,1),r[0]=A.b,r[1]=A.a,r[2]=e.a,r[3]=e.b,P=n.el(t.R,r),d.Math.abs(P)<.3||d.Math.abs(P)>G8?3:P<0?2:1;},n.pf=function(t,A,e){var r,P,i,D,u;return !n.Vp(t.R,A)||(r=n.Zl(t.R,0,A),P=n.Zl(t.R,1,A),D=new n.lk(t.R,t.f,r,P),D.f&&e==1)||(u=new n.lk(t.R,t.f,P,r),u.f&&e==1)||D.f&&u.f?false:(e==3&&(D.f&&(t.V[A]=n.xg(t,P)),u.f&&(t.V[A]=n.xg(t,r))),i=t.jb?n.rf(t,D,u):n.qf(D,u),e==1?t.o[A]=i:e==2&&(D.f&&(i==2?n.bh(t.e[D.b],Mt(t.f[P])):n.bh(t.e[D.d],Mt(t.f[P]))),u.f&&(i==2?n.bh(t.e[u.b],Mt(t.f[r])):n.bh(t.e[u.d],Mt(t.f[r])))),true);},n.qf=function(t,A){var e,r,P;if(r=n.kk(t),P=n.kk(A),r==-1||P==-1||(r+P&1)==0)return 3;switch(e=0,r+P){case 3:case 7:e=1;break;case 5:e=2;}return e;},n.rf=function(t,A,e){var r,P;return r=B(Q.S7,H,6,4,15,1),r[0]=A.b,r[1]=A.a,r[2]=e.a,r[3]=e.b,P=n.el(t.R,r),d.Math.abs(P)<.3||d.Math.abs(P)>G8?3:P<0?1:2;},n.sf=function(t,A,e){var r,P,i,D,u,o,l,U;if(t.o[A]!=0)return  false;if(t.R.K){for(l=0;l<2;l++)for(r=n.Zl(t.R,l,A),U=0;U<n.Lo(t.R,r);U++)if(n.Jm(t.R,n.Zo(t.R,r,U)))return  false;}return n.fm(t.R,A)==1?n.pf(t,A,e):n.fm(t.R,A)!=2||n.Up(t.R,A)||(i=n.Zl(t.R,0,A),D=n.Zl(t.R,1,A),n.$o(t.R,i)==1||n.$o(t.R,D)==1)||n.$o(t.R,i)>3||n.$o(t.R,D)>3||n.Qo(t.R,i)==2||n.Qo(t.R,D)==2||(u=new n.lk(t.R,t.f,D,i),u.f&&e==1)||(o=new n.lk(t.R,t.f,i,D),o.f&&e==1)||u.f&&o.f?false:(e==3&&(u.f&&u.c&&(t.V[A]=true),o.f&&o.c&&(t.V[A]=true)),P=n.Gm(t.R,A)?3:t.jb?n.uf(t,u,o):n.tf(u,o),e==1?t.o[A]=P:e==2&&(u.f&&(P==1?n.bh(t.e[u.b],Mt(t.f[i])):P==2&&n.bh(t.e[u.d],Mt(t.f[i]))),o.f&&(P==1?n.bh(t.e[o.b],Mt(t.f[D])):P==2&&n.bh(t.e[o.d],Mt(t.f[D])))),true);},n.tf=function(t,A){return n.kk(t)==-1||n.kk(A)==-1||((n.kk(t)|n.kk(A))&1)!=0?3:n.kk(t)==n.kk(A)?1:2;},n.uf=function(t,A,e){var r,P,i,D,u,o,l;return i=B(Q.Q7,jt,6,3,15,1),i[0]=n.Ql(t.R,e.a)-n.Ql(t.R,A.a),i[1]=n.Rl(t.R,e.a)-n.Rl(t.R,A.a),i[2]=n.Sl(t.R,e.a)-n.Sl(t.R,A.a),o=B(Q.Q7,jt,6,3,15,1),o[0]=n.Ql(t.R,A.b)-n.Ql(t.R,A.a),o[1]=n.Rl(t.R,A.b)-n.Rl(t.R,A.a),o[2]=n.Sl(t.R,A.b)-n.Sl(t.R,A.a),l=B(Q.Q7,jt,6,3,15,1),l[0]=n.Ql(t.R,e.b)-n.Ql(t.R,e.a),l[1]=n.Rl(t.R,e.b)-n.Rl(t.R,e.a),l[2]=n.Sl(t.R,e.b)-n.Sl(t.R,e.a),D=B(Q.Q7,jt,6,3,15,1),D[0]=i[1]*o[2]-i[2]*o[1],D[1]=i[2]*o[0]-i[0]*o[2],D[2]=i[0]*o[1]-i[1]*o[0],u=B(Q.Q7,jt,6,3,15,1),u[0]=i[1]*D[2]-i[2]*D[1],u[1]=i[2]*D[0]-i[0]*D[2],u[2]=i[0]*D[1]-i[1]*D[0],r=(o[0]*u[0]+o[1]*u[1]+o[2]*u[2])/(d.Math.sqrt(o[0]*o[0]+o[1]*o[1]+o[2]*o[2])*d.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),P=(l[0]*u[0]+l[1]*u[1]+l[2]*u[2])/(d.Math.sqrt(l[0]*l[0]+l[1]*l[1]+l[2]*l[2])*d.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),r<0^P<0?1:2;},n.vf=function(t,A){var e,r,P,i;return e=n.zl(t.R,A),r=n.mp(t.R,A,false),P=n.mp(t.R,A,true),i=-1,r!=P?e!=-1&&e>r?i=e<<24>>24:i=r<<24>>24:e!=-1?(e>P||e<P&&e>=n.vp(t.R,A))&&(i=e<<24>>24):!n.Aq(t.R,A)&&n.cp(t.R,A)!=0&&(i=n.vp(t.R,A),i-=n.mm(t.R,A,i)),n.Rf(t,A,i),i;},n.wf=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(r=B(Q.S7,H,6,t.O,15,1),A=0;A<t.R.f;A++){for(U=n.$o(t.R,A)+n.rp(t.R,A),l=0,i=0;i<n.Mo(t.R,A);i++)if(i<n.$o(t.R,A)||i>=n.Lo(t.R,A)){for(y=2*t.f[n.Zo(t.R,A,i)],e=n._o(t.R,A,i),n.fm(t.R,e)==2&&(n.Up(t.R,e)||++y),u=0;u<l&&!(y<r[u]);u++);for(o=l;o>u;o--)r[o]=r[o-1];r[u]=y,++l;}for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),D=U;D<t.O;D++)n.ah(t.e[A],t.c+1,0);for(P=0;P<U;P++)n.ah(t.e[A],t.c+1,Q.Mfb(r[P]));}},n.xf=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g;if(t.db[A]!=0||n.Tl(t.R,A)!=5&&n.Tl(t.R,A)!=6&&n.Tl(t.R,A)!=7&&n.Tl(t.R,A)!=14&&n.Tl(t.R,A)!=15&&n.Tl(t.R,A)!=16||(t.Q&Z1)!=0&&n.Tl(t.R,A)>=14)return  false;if(n.Qo(t.R,A)!=0){if(n.Wp(t.R,A))return n.mf(t,A,e);if(n.Tl(t.R,A)!=15&&n.Tl(t.R,A)!=16)return  false;}if(n.$o(t.R,A)<3||n.Lo(t.R,A)>4||n.Bl(t.R,A)>0&&n.Tl(t.R,A)==6||n.Tl(t.R,A)==5&&n.Lo(t.R,A)!=4)return  false;if(t.R.K){if(n.Jm(t.R,A))return  false;for(u=0;u<n.Lo(t.R,A);u++)if(n.Jm(t.R,n.Zo(t.R,A,u)))return  false;}if(n.Tl(t.R,A)==7&&!t.S[A])return  false;for(b=B(Q.S7,H,6,4,15,1),g=B(Q.S7,H,6,4,15,1),U=B(Q.xfb,Tt,6,4,16,1),o=0;o<n.Lo(t.R,A);o++){for(i=-1,P=0,l=0;l<n.Lo(t.R,A);l++)U[l]||i<t.f[n.Zo(t.R,A,l)]&&(i=t.f[n.Zo(t.R,A,l)],P=l);b[o]=P,g[o]=i,U[P]=true;}if(n.Lo(t.R,A)==4&&g[0]==g[1]&&g[2]==g[3]||n.Lo(t.R,A)==4&&(g[0]==g[2]||g[1]==g[3])||n.Lo(t.R,A)==3&&g[0]==g[2])return  false;for(y=0,T=0,c=false,D=1;D<n.Lo(t.R,A);D++)if(g[D-1]==g[D]){if(e==1||g[D]==0)return  false;y=n.Zo(t.R,A,b[D-1]),T=n.Zo(t.R,A,b[D]),e==3&&n.gq(t.R,n._o(t.R,A,b[D]))&&(t.W[A]=true),c=true;}return e!=1&&!c?false:(r=t.jb?n.zf(t,A,b):n.yf(t,A,b),e==1?t.db[A]=r:e==2&&(r==1?n.bh(t.e[y],Mt(t.f[A])):r==2&&n.bh(t.e[T],Mt(t.f[A]))),true);},n.yf=function(t,A,e){var r,P,i,D,u,o,l,U,y,T;for(T=j(v(Q.S7,2),E1,7,0,[j(v(Q.S7,1),H,6,15,[2,1,2,1]),j(v(Q.S7,1),H,6,15,[1,2,2,1]),j(v(Q.S7,1),H,6,15,[1,1,2,2]),j(v(Q.S7,1),H,6,15,[2,1,1,2]),j(v(Q.S7,1),H,6,15,[2,2,1,1]),j(v(Q.S7,1),H,6,15,[1,2,1,2])]),r=B(Q.Q7,jt,6,n.Lo(t.R,A),15,1),D=0;D<n.Lo(t.R,A);D++)r[D]=n.Yl(t.R,n.Zo(t.R,A,e[D]),A);if(l=n.ep(t.R,A,e,r,null)<<24>>24,l!=3)return l;for(U=0,y=0,u=0;u<n.Lo(t.R,A);u++)P=n._o(t.R,A,e[u]),n.Zl(t.R,0,P)==A&&(n.im(t.R,P)==129&&(y!=0&&n.Vn(t.R,A),U=u,y=1),n.im(t.R,P)==257&&(y!=0&&n.Vn(t.R,A),U=u,y=2));if(y==0)return 3;for(i=1;i<n.Lo(t.R,A);i++)r[i]<r[0]&&(r[i]+=N1);if(n.Lo(t.R,A)==3){switch(U){case 0:(r[1]<r[2]&&r[2]-r[1]<t1||r[1]>r[2]&&r[1]-r[2]>t1)&&(y=3-y);break;case 1:r[2]-r[0]>t1&&(y=3-y);break;case 2:r[1]-r[0]<t1&&(y=3-y);}return y==1?2:1;}return o=0,r[1]<=r[2]&&r[2]<=r[3]?o=0:r[1]<=r[3]&&r[3]<=r[2]?o=1:r[2]<=r[1]&&r[1]<=r[3]?o=2:r[2]<=r[3]&&r[3]<=r[1]?o=3:r[3]<=r[1]&&r[1]<=r[2]?o=4:r[3]<=r[2]&&r[2]<=r[1]&&(o=5),T[o][U]==y?2:1;},n.zf=function(t,A,e){var r,P,i,D,u,o;for(r=B(Q.S7,H,6,4,15,1),u=0;u<n.Lo(t.R,A);u++)r[u]=n.Zo(t.R,A,e[u]);for(n.Lo(t.R,A)==3&&(r[3]=A),P=T1(Q.Q7,[Q1,jt],[15,6],15,[3,3],2),D=0;D<3;D++)P[D][0]=n.Ql(t.R,r[D+1])-n.Ql(t.R,r[0]),P[D][1]=n.Rl(t.R,r[D+1])-n.Rl(t.R,r[0]),P[D][2]=n.Sl(t.R,r[D+1])-n.Sl(t.R,r[0]);return o=B(Q.Q7,jt,6,3,15,1),o[0]=P[0][1]*P[1][2]-P[0][2]*P[1][1],o[1]=P[0][2]*P[1][0]-P[0][0]*P[1][2],o[2]=P[0][0]*P[1][1]-P[0][1]*P[1][0],i=(P[2][0]*o[0]+P[2][1]*o[1]+P[2][2]*o[2])/(d.Math.sqrt(P[2][0]*P[2][0]+P[2][1]*P[2][1]+P[2][2]*P[2][2])*d.Math.sqrt(o[0]*o[0]+o[1]*o[1]+o[2]*o[2])),i>0?1:2;},n.Af=function(t){var A,e;for(A=0,s.Vrb(t.e),e=0;e<t.e.length;e++)(e==0||n.dh(t.e[e],t.e[e-1])!=0)&&++A,t.f[t.e[e].a]=A;return A;},n.Bf=function(t){var A,e,r,P,i,D,u,o,l,U,y,T;if(!t.A){for(t.A=new s.Mj(),U=0,y=B(Q.S7,H,6,t.R.f,15,1),D=B(Q.S7,H,6,t.R.f,15,1),o=B(Q.S7,H,6,t.R.g,15,1),A=0;A<t.R.f;A++)if(y[A]==0&&(n.fq(t.R,A)||n.Qo(t.R,A)==1)){for(D[0]=A,u=1,l=0,y[A]=++U,e=B(Q.xfb,Tt,6,t.R.g,16,1),i=0;i<u;i++)for(T=0;T<n.$o(t.R,D[i]);T++)P=n._o(t.R,D[i],T),(n.gq(t.R,P)||n.fm(t.R,P)==2||n.Vp(t.R,P))&&(r=n.Zo(t.R,D[i],T),e[P]||(o[l++]=P,e[P]=true),y[r]==0&&(D[u++]=r,y[r]=U));s.sj(t.A,new n.kh(D,u,o,l));}}},n.Cf=function(t){var A;for(t.S=B(Q.xfb,Tt,6,t.R.f,16,1),A=0;A<t.R.f;A++)if(n.Tl(t.R,A)==7){if(n.$o(t.R,A)==4){t.S[A]=true;continue;}if(n.$o(t.R,A)==3){if(n.Uo(t.R,A)==3){t.S[A]=true;continue;}if(n.Bl(t.R,A)==1){t.S[A]=true;continue;}if(n.$p(t.R,A,true))continue;if((t.Q&32)!=0){t.S[A]=true;continue;}if(n.eq(t.R,A)){t.S[A]=true;continue;}}}},n.Df=function(t,A){var e,r,P,i;for(P=false,r=0;r<t.R.g;r++)n.sf(t,r,1)&&(t.q[r]=t.L,A&&n.Sf(t,r),P=true);for(i=false,e=0;e<t.R.f;e++)n.xf(t,e,1)&&(t.ib[e]=t.L,A&&n.Tf(t,e),i=true);return i&&(t.L=!t.L),P||i;},n.Ef=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J;for(S=B(Q.xfb,Tt,6,t.R.f,16,1),R=B(Q.xfb,Tt,6,t.R.g,16,1),A=0,O=false,(t.Q&128)!=0&&(t.Y=B(Q.S7,H,6,t.R.f,15,1),t.X=B(Q.S7,H,6,t.R.g,15,1)),r=0;r<t.R.f;r++)t.W[r]&&(t.fb[r]||n.xf(t,r,1)&&(t.fb[r]=true,S[r]=true,++A));for(i=0;i<t.R.g;i++)t.V[i]&&(t.p[i]||n.sf(t,i,1)&&(t.p[i]=true,R[i]=true,++A));if(A==1){for(e=0;e<t.R.f;e++)if(S[e]){t.db[e]=0;break;}for(P=0;P<t.R.g;P++)if(R[P]){t.o[P]=0;break;}}else if(A>1)for(n.Bf(t),t.T=0,u=new s.trb(t.A);u.a<u.c.a.length;){for(D=s.srb(u),E=0,x=0,U=0,l=0,y=-1,o=-1,b=0;b<D.a.length;b++)S[D.a[b]]&&(++E,(t.db[D.a[b]]==1||t.db[D.a[b]]==2)&&(++x,O=true,y<t.f[D.a[b]]&&(y=t.f[D.a[b]],U=D.a[b])));for(g=0;g<D.b.length;g++)R[D.b[g]]&&(++E,G=t.f[n.Zl(t.R,0,D.b[g])],J=t.f[n.Zl(t.R,1,D.b[g])],T=G>J?(G<<16)+J:(J<<16)+G,(t.o[D.b[g]]==1||t.o[D.b[g]]==2)&&(++x,O=true,o<T&&(o=T,l=D.b[g])));if(E!=0)if(E==1){for(k=0;k<D.a.length;k++)S[D.a[k]]&&(t.db[D.a[k]]=0);for(c=0;c<D.b.length;c++)R[D.b[c]]&&(t.o[D.b[c]]=0);}else if(x==1){for(k=0;k<D.a.length;k++)S[D.a[k]]&&(t.db[D.a[k]]=3);for(c=0;c<D.b.length;c++)R[D.b[c]]&&(t.o[D.b[c]]=3);}else {if((t.Q&128)!=0){for(++t.T,k=0;k<D.a.length;k++)S[D.a[k]]&&(t.Y[D.a[k]]=t.T);for(c=0;c<D.b.length;c++)R[D.b[c]]&&(t.X[D.b[c]]=t.T);}if(F=false,y!=-1?t.db[U]==2&&(F=true):t.o[l]==2&&(F=true),F){for(k=0;k<D.a.length;k++)if(S[D.a[k]])switch(t.db[D.a[k]]){case 1:t.db[D.a[k]]=2;break;case 2:t.db[D.a[k]]=1;}for(c=0;c<D.b.length;c++)if(R[D.b[c]])switch(t.o[D.b[c]]){case 1:t.o[D.b[c]]=2;break;case 2:t.o[D.b[c]]=1;}}}}return O;},n.Ff=function(t,A){var e,r,P,i,D,u,o,l,U,y,T;for(i=T1(Q.S7,[E1,H],[7,6],15,[2,32],2),D=0;D<2;D++){for(e=B(Q.S7,E1,7,32,0,2),T=0,P=0;P<32;P++)if(A[D][P]!=null){for(U=A[D][P].length,e[P]=B(Q.S7,H,6,U,15,1),u=0;u<U;u++)e[P][u]=t.f[A[D][P][u]];w.nAb(e[P],w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),++T;}for(y=T;y>0;y--){for(l=0,o=null,r=0;r<32;r++)if(e[r]!=null){if(o==null||o.length<e[r].length)o=e[r],l=r;else if(o.length==e[r].length){for(u=o.length-1;u>=0;u--)if(o[u]<e[r][u]){o=e[r],l=r;break;}}}i[D][l]=y,e[l]=null;}}return i;},n.Gf=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R;if(y=false,t.R.K){for(l=0;l<t.R.g;l++)if(n.hm(t.R,l)!=0){y=true;break;}}for(t.O=2,e=0;e<t.R.f;e++)t.O=d.Math.max(t.O,n.$o(t.R,e)+n.rp(t.R,e));for(o=d.Math.max(2,y?(62+t.c+t.O*(t.c+23))/63|0:(62+t.c+t.O*(t.c+5))/63|0),t.f=B(Q.S7,H,6,t.R.q,15,1),t.e=B(Q.c8,Hn,101,t.R.f,0,1),r=0;r<t.R.f;r++)t.e[r]=new n.fh(o);for(u=false,P=0;P<t.R.f;P++)n.eh(t.e[P],P),Rt(z(n.Ol(t.R,P),1),0)||n.Jl(t.R,P)!=null?n.ah(t.e[P],8,6):n.ah(t.e[P],8,Mt(n.Tl(t.R,P))),n.ah(t.e[P],8,Mt(n.Ml(t.R,P))),n.ah(t.e[P],2,Mt(n.Qo(t.R,P))),n.ah(t.e[P],4,Mt(n.$o(t.R,P)+n.rp(t.R,P))),Rt(z(n.Ol(t.R,P),1),0)?n.ah(t.e[P],4,8):n.ah(t.e[P],4,Mt(8+n.Bl(t.R,P))),n.ah(t.e[P],5,Mt(d.Math.min(31,n.Uo(t.R,P)))),n.ah(t.e[P],4,Mt(n.vf(t,P)+1)),n.ah(t.e[P],2,Mt(n.Pl(t.R,P)>>4)),t.R.K&&(n.ah(t.e[P],51,n.Ol(t.R,P)),n.Jl(t.R,P)!=null&&(u=true));if(t.U=n.Kf(t),t.U<t.R.f){for(A=0;A<t.R.f;A++){for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),T=B(Q.S7,H,6,n.$o(t.R,A),15,1),b=0;b<n.$o(t.R,A);b++)T[b]=t.f[n.Zo(t.R,A,b)]<<5,T[b]|=d.Math.min(31,n.Yo(t.R,n._o(t.R,A,b)));for(w.nAb(T,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),g=t.O;g>T.length;g--)n.ah(t.e[A],t.c+5,0);for(c=T.length-1;c>=0;c--)n.ah(t.e[A],t.c+5,Q.Mfb(T[c]));}t.U=n.Kf(t);}if(u&&t.U<t.R.f){for(A=0;A<t.R.f;A++){for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),D=n.Jl(t.R,A),S=D==null?0:d.Math.min(12,D.length),b=12;b>S;b--)n.ah(t.e[A],8,0);for(c=S-1;c>=0;c--)n.ah(t.e[A],8,Q.Mfb(D[c]));}t.U=n.Kf(t);}if(y&&t.U<t.R.f){for(A=0;A<t.R.f;A++){for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),U=B(Q.T7,e0,6,n.$o(t.R,A)+n.rp(t.R,A),14,1),k=0,b=0;b<n.Mo(t.R,A);b++)(b<n.$o(t.R,A)||b>=n.Lo(t.R,A))&&(U[k]=Mt(t.f[n.Zo(t.R,A,b)]),U[k]=c1(U[k],23),U[k]=Pt(U[k],Mt(n.hm(t.R,n._o(t.R,A,b)))),++k);for(w.nAb(U,w.UAb(T0(s.asb.prototype.md,s.asb,[]))),g=t.O;g>U.length;g--)n.ah(t.e[A],t.c+23,0);for(c=U.length-1;c>=0;c--)n.ah(t.e[A],t.c+23,U[c]);}t.U=n.Kf(t);}if((t.Q&8)!=0&&t.U<t.R.f){for(F=new n.Zv(),i=0;i<t.R.f;i++)n.El(t.R,i)!=null&&n.Xv(F,n.El(t.R,i));for(A=0;A<t.R.f;A++)R=n.El(t.R,A)==null?0:1+n.Yv(F,n.El(t.R,A)),n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),n.ah(t.e[A],t.c,Q.Mfb(R));t.U=n.Kf(t);}if((t.Q&16)!=0&&t.U<t.R.f){for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),n.ah(t.e[A],1,Mt(n.Pm(t.R,A)?1:0));t.U=n.Kf(t);}(t.Q&512)!=0&&t.R.K&&n.jf(t);},n.Hf=function(t){var A,e,r,P,i,D,u,o,l;for(e=0;e<t.R.f;e++)n.eh(t.e[e],e),n.ah(t.e[e],2*t.c+4,c1(Mt(t.f[e]),t.c+4));for(o=1;o<=t.U;o++){for(P=false,r=0;r<t.R.f;r++)t.f[r]==o&&(P=P|n.xf(t,r,2));if(P){if(u=t.U,t.U=n.Kf(t),t.U!=u)return  true;for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],2*t.c+4,c1(Mt(t.f[A]),t.c+4));}}for(l=B(Q.d8,Hn,173,t.R.g,0,1),D=0;D<l.length;D++)l[D]=new n.ih(t.f[n.Zl(t.R,0,D)],t.f[n.Zl(t.R,1,D)],D);for(s.Trb(l,0,l.length,null),i=0;i<l.length;i++)if(n.sf(t,l[i].a,2)){for(;i+1<l.length&&n.hh(l[i],l[i+1])==0;)n.sf(t,l[++i].a,2);if(u=t.U,t.U=n.Kf(t),t.U!=u)return  true;for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],2*t.c+4,c1(Mt(t.f[A]),t.c+4));}return  false;},n.If=function(t){var A,e,r,P,i,D,u,o,l,U;for(i=0,e=0;e<t.R.f;e++)t.bb[e]!=0&&(t.bb[e]!=2||(t.Q&256)==0)&&++i;if(i!=0){for(U=B(Q.S7,H,6,i,15,1),i=0,r=0;r<t.R.f;r++)t.bb[r]!=0&&(t.bb[r]!=2||(t.Q&256)==0)&&(U[i]=t.bb[r]<<29|t.ab[r]<<24|t.f[r]<<12|r,++i);for(w.nAb(U,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),D=0,l=0,u=U[0]&G0;;)if(++l,l==U.length||u!=(U[l]&G0)){for(P=B(Q.S7,H,6,l-D,15,1),o=D;o<l;o++)A=U[o]&4095,P[o-D]=A,t.gb[A]=true;if(s.sj(t.hb,P),l==U.length)break;u=U[l]&G0,D=l;}}},n.Jf=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c;for(u=false,o=0;o<t.hb.a.length;o++){for(D=s.xj(t.hb,o),A=true,c=-1,l=false,y=0;y<D.length;y++){if(e=D[y],t.db[e]==0){A=false;break;}if(t.db[e]!=3){for(U=true,T=0;T<D.length;T++)if(T!=y&&t.f[e]==t.f[D[T]]){U=false;break;}U&&c<t.f[e]&&(c=t.f[e],l=t.db[e]==1);}}if(A&&c!=-1){for(r=D,P=0,i=r.length;P<i;++P)e=r[P],(t.db[e]==1||t.db[e]==2)&&(t.eb[e]=l),t.gb[e]=false;s.Fj(t.hb,D),u=true,--o;}}return u;},n.Kf=function(t){var A,e;A=n.Af(t);do e=A,n.wf(t),A=n.Af(t);while(e!=A);return A;},n.Lf=function(t){for((t.Q&1)!=0&&(t.Q&2)==0&&(t.g=s.Brb(t.f,t.R.f)),(t.Q&B1)==0&&(t.W=B(Q.xfb,Tt,6,t.R.f,16,1),t.V=B(Q.xfb,Tt,6,t.R.g,16,1),t.U<t.R.f&&(n.kf(t),(t.Q&B1)==0&&(n.Jf(t),t.P&&n.uh(t.P,t.f)))),t.g==null&&(t.Q&1)!=0&&(t.Q&2)!=0&&(t.g=s.Brb(t.f,t.R.f));t.U<t.R.f;)n.lf(t),(t.Q&B1)==0&&(n.Jf(t),t.P&&n.uh(t.P,t.f));(t.Q&B1)==0&&(n.Jf(t),n.Ef(t),n.hg(t));},n.Mf=function(t){var A,e,r,P,i;for(i=t.U,P=s.Brb(t.f,t.R.f),t.R.K||(n.Of(t),n.Cg(t,i,P)),t.bb=B(Q.O7,Y,6,t.R.f,15,1),t.ab=B(Q.O7,Y,6,t.R.f,15,1),e=0;e<t.R.f;e++)t.bb[e]=n.Hl(t.R,e)<<24>>24,t.ab[e]=n.Gl(t.R,e)<<24>>24;for(t.n=B(Q.O7,Y,6,t.R.g,15,1),t.k=B(Q.O7,Y,6,t.R.g,15,1),r=0;r<t.R.g;r++)t.n[r]=n.cm(t.R,r)<<24>>24,t.k[r]=n.bm(t.R,r)<<24>>24;for(n.Nf(t),t.Z=false,t.M=B(Q.xfb,Tt,6,t.R.f,16,1),A=0;A<t.R.f;A++)t.db[A]!=0&&(t.M[A]=true,t.Z=true);n.Qf(t),t.P=null,t.cb=B(Q.xfb,Tt,6,t.R.f,16,1),t.Z&&(t.P=new n.Ah(t.R,P,t.M,t.db,t.o,t.bb,t.ab,t.ib,t.q,t.cb),n.vh(t.P)),t.eb=B(Q.xfb,Tt,6,t.R.f,16,1),t.gb=B(Q.xfb,Tt,6,t.R.f,16,1),t.hb=new s.Mj(),t.P&&n.uh(t.P,t.f),n.If(t),n.Cg(t,i,P),n.Pf(t),t.P&&(t.K=n.rh(t.P)),n.$f(t);},n.Nf=function(t){var A,e,r,P,i,D;for(t.L=true,i=n.Df(t,false);t.U<t.R.f&&i;){for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),D=t.db[A]<<7,(t.db[A]==1||t.db[A]==2)&&t.bb[A]!=0&&(D|=t.bb[A]<<5,D|=t.ab[A]),n.ah(t.e[A],18,c1(Q.Mfb(D),9));for(e=0;e<t.R.g;e++)r=t.o[e]<<7,(t.o[e]==1||t.o[e]==2)&&n.im(t.R,e)==1&&t.n[e]!=0&&(r|=t.n[e]<<5,r|=t.k[e]),n.bh(t.e[n.Zl(t.R,0,e)],Q.Mfb(r)),n.bh(t.e[n.Zl(t.R,1,e)],Q.Mfb(r));if(P=n.Kf(t),t.U==P)break;t.U=P,i=n.Df(t,false);}},n.Of=function(t){var A,e,r,P;for(t.L=true,t.$=B(Q.O7,Y,6,t.R.f,15,1),t.i=B(Q.O7,Y,6,t.R.g,15,1),P=n.Df(t,true);t.U<t.R.f&&P;){for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c+4,Pt(c1(Mt(t.f[A]),4),Mt(t.db[A]<<2)));for(e=0;e<t.R.g;e++)n.bh(t.e[n.Zl(t.R,0,e)],Mt(t.o[e])),n.bh(t.e[n.Zl(t.R,1,e)],Mt(t.o[e]));if(r=n.Kf(t),t.U==r)break;t.U=r,P=n.Df(t,true);}},n.Pf=function(t){var A,e,r,P,i,D,u;for(t.L=true,r=n.Zf(t),t.P&&n.uh(t.P,t.f)&&(r=n.Zf(t)),n.Df(t,false)&&n.Jf(t),D=true;t.U<t.R.f&&D;){for(P=n.Ff(t,r),A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c,Mt(t.f[A])),n.ah(t.e[A],20,0),!t.cb[A]&&t.bb[A]!=0&&n.bh(t.e[A],v1(Mt(t.bb[A]<<18),c1(Mt(P[t.bb[A]==1?0:1][t.ab[A]]),8))),u=t.db[A],t.eb[A]&&(u==1?u=2:u==2&&(u=1)),n.bh(t.e[A],Q.Mfb(u<<4));for(e=0;e<t.R.g;e++)n.bh(t.e[n.Zl(t.R,0,e)],Mt(t.o[e])),n.bh(t.e[n.Zl(t.R,1,e)],Mt(t.o[e]));if(i=n.Kf(t),t.U==i)break;t.U=i,D=false,t.P&&n.uh(t.P,t.f)&&(D=true,r=n.Zf(t)),n.Df(t,false)&&(D=true,n.Jf(t));}},n.Qf=function(t){var A,e;for(A=0;A<t.R.f;A++)(!t.M[A]||t.db[A]==3)&&(t.bb[A]=0);for(e=0;e<t.R.g;e++)(n.im(t.R,e)!=1||t.o[e]==0||t.o[e]==3)&&(t.n[e]=0);},n.Rf=function(t,A,e){t.a==null&&(t.a=B(Q.O7,Y,6,t.R.f,15,1),s.Erb(t.a)),t.a[A]=e<<24>>24;},n.Sf=function(t,A){var e,r,P,i,D,u;if((t.o[A]==1||t.o[A]==2)&&!n.jq(t.R,A)){D=false;try{for(i=0;i<2;i++)if(e=n.Zl(t.R,i,A),n.$o(t.R,e)==3){for(r=B(Q.S7,H,6,2,15,1),P=0,u=0;u<n.$o(t.R,e);u++)n._o(t.R,e,u)!=A&&(r[P++]=n.Zo(t.R,e,u));t.f[r[0]]>t.f[r[1]]^n.Uf(t,e,r[0],r[1])&&(D=!D);}}catch(o){if(o=M1(o),Zt(o,19)){t.i[A]=3;return;}else throw Lt(o);}t.o[A]==1^D?t.i[A]=1:t.i[A]=2;}},n.Tf=function(t,A){var e,r,P,i,D,u,o;if(t.db[A]==1||t.db[A]==2){if(u=false,n.Qo(t.R,A)==2&&n.$o(t.R,A)==2)try{for(D=0;D<2;D++)if(e=n.Zo(t.R,A,D),n.$o(t.R,e)==3){for(P=B(Q.S7,H,6,2,15,1),i=0,o=0;o<n.$o(t.R,e);o++)n.ap(t.R,e,o)==1&&(P[i++]=n.Zo(t.R,e,o));t.f[P[0]]>t.f[P[1]]^n.Uf(t,e,P[0],P[1])&&(u=!u);}}catch(l){if(l=M1(l),Zt(l,19)){t.$[A]=3;return;}else throw Lt(l);}else {try{r=n.Wf(t,A);}catch(l){if(l=M1(l),Zt(l,19)){t.$[A]=3;return;}else throw Lt(l);}for(D=1;D<r.length;D++)for(o=0;o<D;o++)t.f[r[D]]<t.f[r[o]]&&(u=!u);}t.db[A]==1^u?t.$[A]=1:t.$[A]=2;}},n.Uf=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt;if(n.Tl(t.R,e)!=n.Tl(t.R,r))return n.Tl(t.R,e)>n.Tl(t.R,r);if(n.Ml(t.R,e)!=n.Ml(t.R,r))return it=n.Mm(t.R,e)?(n.Rk(),n.Pk)[n.Tl(t.R,e)]:n.Ml(t.R,e),lt=n.Mm(t.R,r)?(n.Rk(),n.Pk)[n.Tl(t.R,r)]:n.Ml(t.R,r),it>lt;for(x=t.R.f,S=B(Q.S7,H,6,x,15,1),E=B(Q.S7,H,6,x,15,1),O=B(Q.S7,H,6,x,15,1),R=B(Q.xfb,Tt,6,x,16,1),o=B(Q.xfb,Tt,6,t.R.q,16,1),S[0]=A,S[1]=e,S[2]=r,E[0]=-1,E[1]=0,E[2]=0,o[A]=true,o[e]=true,o[r]=true,T=1,G=2,K=B(Q.S7,H,6,64,15,1),K[1]=1,K[2]=3,b=2;T<=G;){for(;T<K[b];){if(c=S[T],!R[T]){for(g=0,k=0,V=0;V<n.$o(t.R,c);V++){if(U=n.Zo(t.R,c,V),G+n.ap(t.R,c,V)+1>=x&&(x+=t.R.f,S=n.Gg(S,x),E=n.Gg(E,x),O=n.Gg(O,x),R=n.Hg(R,x)),n.Yp(t.R,n._o(t.R,c,V)))++g,k+=n.Tl(t.R,U);else if(U!=A)for(ot=1;ot<n.ap(t.R,c,V);ot++)++G,S[G]=U,E[G]=T,R[G]=true;if(Dt=E[T],U!=S[Dt]){if(u=false,o[U])for(ut=E[Dt];ut!=-1;){if(U==S[ut]){u=true;break;}ut=E[ut];}u?(++G,S[G]=U,E[G]=T,R[G]=true):(++G,S[G]=U,E[G]=T,o[U]=true);}}g!=0&&(++G,O[G]=(k<<2)/g|0,E[G]=T,R[G]=true);}if(++T,T==Z0)throw Lt(new h.cC("Emergency break in while loop."));}for(K.length==b+1&&(K=n.Gg(K,K.length+64)),K[b+1]=G+1,J=K[b];J<K[b+1];J++)O[J]==0&&(O[J]=(n.Tl(t.R,S[J])==151||n.Tl(t.R,S[J])==152?1:n.Tl(t.R,S[J]))<<2),O[J]+=O[E[J]]<<16;if(n.Yf(t,R,O,E,S,K,b),O[1]!=O[2])return O[1]>O[2];b>1&&n.Vf(O,E,K,b),++b;}for(y=B(Q.S7,H,6,t.R.f,15,1),At=false,i=0;i<t.R.f;i++)if(o[i]&&!n.Mm(t.R,i)){At=true;break;}if(At){for(D=0;D<t.R.f;D++)y[D]=n.Mm(t.R,D)?(n.Rk(),n.Pk)[n.Tl(t.R,D)]:n.Ml(t.R,D);if(n.Xf(t,R,O,E,S,y,K,b))return O[1]>O[2];}for(s.Lrb(y,y.length,0),F=false,l=0;l<t.R.g;l++)(o[n.Zl(t.R,0,l)]||o[n.Zl(t.R,1,l)])&&(t.i[l]==1?(y[n.Zl(t.R,0,l)]=1,y[n.Zl(t.R,1,l)]=1,F=true):t.i[l]==2&&(y[n.Zl(t.R,0,l)]=2,y[n.Zl(t.R,1,l)]=2,F=true));if(F&&n.Xf(t,R,O,E,S,y,K,b))return O[1]>O[2];for(s.Lrb(y,y.length,0),Qt=false,P=0;P<t.R.f;P++)o[P]&&(t.$[P]==2?(y[P]=1,Qt=true):t.$[P]==1&&(y[P]=2,Qt=true));if(Qt&&n.Xf(t,R,O,E,S,y,K,b))return O[1]>O[2];throw t.d=true,Lt(new h.cC("no distinction applying CIP rules"));},n.Vf=function(t,A,e,r){var P,i,D,u,o,l,U,y,T;for(y=e[r],D=e[r+1]-y,T=B(Q.a8,C1,175,D,0,1),o=0;o<D;o++)T[o]=new n._g(),T[o].c=t[o+y],T[o].b=A[o+y],T[o].a=o+y;for(P=new n.Yg(),U=r;U>1;U--){for(l=0;l<D;l++)T[l].c+=t[T[l].b]<<16,T[l].b=A[T[l].b];for(s.Trb(T,0,T.length,P),i=1,u=0;u<D;u++)t[T[u].a]=i,u!=D-1&&n.Xg(T[u],T[u+1])!=0&&++i;}},n.Wf=function(t,A){var e,r,P,i,D,u,o;for(D=n.Lo(t.R,A),u=B(Q.S7,H,6,D,15,1),P=0;P<D;P++)u[P]=n.Zo(t.R,A,P);for(r=D;r>1;r--){for(e=false,i=1;i<r;i++)n.Uf(t,A,u[i-1],u[i])&&(e=true,o=u[i-1],u[i-1]=u[i],u[i]=o);if(!e)break;}return u;},n.Xf=function(t,A,e,r,P,i,D,u){var o,l;for(l=1;l<u;l++){for(o=D[l];o<D[l+1];o++)e[o]=i[P[o]]+(e[r[o]]<<8);if(n.Yf(t,A,e,r,P,D,l),e[1]!=e[2])return  true;l>1&&n.Vf(e,r,D,l);}return  false;},n.Yf=function(t,A,e,r,P,i,D){var u,o,l,U,y,T,c,b,g,k,F;for(y=D;y>1;y--){for(g=i[y]-i[y-1],F=B(Q.$7,C1,174,g,0,1),u=i[y],b=0;b<g;b++){for(k=i[y-1]+b,T=u;T<i[y+1]&&r[T]==k;)++T;for(F[b]=new n.Vg(),F[b].c=k,F[b].d=e[k],F[b].b=A[k]?0:n.Cp(t.R,P[k]),F[b].a=B(Q.S7,H,6,T-u,15,1),U=u;U<T;U++)F[b].a[U-u]=e[U];w.nAb(F[b].a,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),u=T;}for(o=new n.Sg(),s.Trb(F,0,F.length,o),l=1,c=0;c<g;c++)e[F[c].c]=l,c!=g-1&&n.Rg(F[c],F[c+1])!=0&&++l;}},n.Zf=function(t){var A,e;for(e=T1(Q.S7,[b0,E1],[28,7],0,[2,32],2),A=0;A<t.R.f;A++)t.M[A]&&(t.bb[A]==1?e[0][t.ab[A]]=n.Bh(e[0][t.ab[A]],A):t.bb[A]==2&&(e[1][t.ab[A]]=n.Bh(e[0][t.ab[A]],A)));return e;},n.$f=function(t){var A,e,r,P,i,D,u,o,l,U,y,T;for(i=0,U=0,D=0,u=0,o=0,l=0,y=0,T=false,A=B(Q.xfb,Tt,6,32,16,1),e=0;e<t.R.f;e++)t.db[e]!=0&&(++i,t.db[e]==3?++U:t.bb[e]==0?(++D,t.P&&n.qh(t.P,e)&&++u):t.bb[e]==2?t.ab[e]==0&&++l:t.bb[e]==1&&(P=t.ab[e],A[P]||(++y,A[P]=true),t.ab[e]==0&&++o,t.P&&n.qh(t.P,e)&&(T=true)));for(r=0;r<t.R.g;r++)t.o[r]!=0&&n.im(t.R,r)==1&&(++i,t.o[r]==3?++U:t.n[r]==0?(++D,t.P&&n.qh(t.P,n.Zl(t.R,0,r))&&n.qh(t.P,n.Zl(t.R,1,r))&&++u):t.n[r]==2?t.k[r]==0&&++l:t.n[r]==1&&(P=t.k[r],A[P]||(++y,A[P]=true),t.k[r]==0&&++o,t.P&&n.qh(t.P,n.Zl(t.R,0,r))&&n.qh(t.P,n.Zl(t.R,1,r))&&(T=true)));if(i==0){n.Nn(t.R,H1);return;}if(U!=0){n.Nn(t.R,0);return;}if(t.K){n.Nn(t.R,r0+(1<<y));return;}o+u==i&&!T?n.Nn(t.R,Ye):D==i?n.Nn(t.R,yA):l==i?n.Nn(t.R,327680):D==i-1&&o==1?n.Nn(t.R,tn):n.Nn(t.R,458752+(1<<y));},n._f=function(t,A,e){for(;e!=0;)t.t==0&&((!t.r||t.v!=63)&&(t.v+=64),h.Gkb(t.u,t.v&n1),t.t=6,t.v=0),t.v<<=1,t.v=zt(Pt(Mt(t.v),z(A,1))),A=A0(A,1),--e,--t.t;},n.ag=function(t){return t.v<<=t.t,(!t.r||t.v!=63)&&(t.v+=64),h.Gkb(t.u,t.v&n1),t.u.a;},n.bg=function(t,A){t.u=new h.Mkb(),t.t=6,t.v=0,t.r=A;},n.cg=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k;if(t.R.f==0){t.s="";return;}for(y=t.jb&t.b,k=t.jb?16:8,n.bg(t,true),h.Gkb(t.u,y?35:33),n._f(t,Mt(t.jb?1:0),1),n._f(t,Q.Mfb(A?1:0),1),n._f(t,Q.Mfb(k>>1),4),b=0,o=1;o<t.R.f;o++)b=n.sg(t,t.B[o],t.F[o]==-1?-1:t.B[t.F[o]],b,e);if(y)for(l=0;l<t.R.f;l++)for(r=t.B[l],c=n.$o(t.R,r);c<n.Lo(t.R,r);c++)b=n.sg(t,n.Zo(t.R,r,c),r,b,e);if(t.R.f>1&&b==0){t.s="";return;}for(D=1<<k,T=b/((D>>1)-1),g=b+.5*T,U=1;U<t.R.f;U++)n.dg(t,t.B[U],t.F[U]==-1?-1:t.B[t.F[U]],g,T,k,e);if(y)for(u=0;u<t.R.f;u++)for(r=t.B[u],c=n.$o(t.R,r);c<n.Lo(t.R,r);c++)n.dg(t,n.Zo(t.R,r,c),r,g,T,k,e);A&&(i=t.jb?1.5:(n.Rk(),n.Rk(),n.Qk),P=n.Xl(t.R,y?t.R.q:t.R.f,y?t.R.r:t.R.g,i,e),n._f(t,Mt(d.Math.min(D-1,d.Math.max(0,pt(.5+d.Math.log(P/.1)*d.Math.LOG10E/(d.Math.log(2e3)*d.Math.LOG10E)*(D-1))))),k),n._f(t,Mt(n.gg(e[t.B[0]].a/P,D)),k),n._f(t,Mt(n.gg(e[t.B[0]].b/P,D)),k),t.jb&&n._f(t,Mt(n.gg(e[t.B[0]].c/P,D)),k)),t.s=n.ag(t);},n.dg=function(t,A,e,r,P,i,D){var u,o,l;u=e==-1?(D[A].a-D[t.B[0]].a)/8:D[A].a-D[e].a,o=e==-1?(D[A].b-D[t.B[0]].b)/8:D[A].b-D[e].b,n._f(t,Mt(pt((r+u)/P)),i),n._f(t,Mt(pt((r+o)/P)),i),t.jb&&(l=e==-1?(D[A].c-D[t.B[0]].c)/8:D[A].c-D[e].c,n._f(t,Mt(pt((r+l)/P)),i));},n.eg=function(t,A){var e;for(e=0;e<t.w;e++)A-=16;for(A<0&&h.Rkb();A>15;)n._f(t,1,1),n._f(t,15,4),A-=16,++t.w;n._f(t,1,1),n._f(t,Q.Mfb(A),4);},n.fg=function(t){var A,e,r,P,i,D;if(t.R.f==0){t.N="";return;}for(i=0,r=false,P=false,e=0;e<t.R.f;e++)i<n.Ll(t.R,e)&&(i=n.Ll(t.R,e)),n.Am(t.R,e)?r=true:P=true;if(i==0){t.N="";return;}for(D=n.Pg(i),n.bg(t,true),n._f(t,Q.Mfb(D),4),n._f(t,Q.Mfb(r?1:0),1),n._f(t,Q.Mfb(P?1:0),1),A=0;A<t.R.f;A++)n._f(t,Mt(n.Ll(t.R,t.B[A])),D),r&&P&&n._f(t,Mt(n.Am(t.R,t.B[A])?1:0),1);t.N=n.ag(t);},n.gg=function(t,A){var e,r,P,i;return e=A/2|0,P=t<0,t=d.Math.abs(t),i=A/32|0,r=d.Math.min(e-1,zt(w1(d.Math.round(t*e/(t+i))))),P?e+r:r;},n.hg=function(t){var A,e,r;for(A=0;A<t.R.f;A++)t.db[A]==3&&!n.wm(t.R,A)&&n.Vn(t.R,A),(n.Hl(t.R,A)==1||n.Hl(t.R,A)==2)&&t.db[A]==3&&n.Vn(t.R,A),n.wm(t.R,A)&&t.db[A]!=3&&!n.Eg(t,A)&&n.Vn(t.R,A);for(r=0;r<t.R.r;r++)n.Rm(t.R,r)&&!n.Dg(t,r)&&n.Vn(t.R,n.Zl(t.R,0,r));for(e=0;e<t.R.g;e++)n.fm(t.R,e)==2&&(n.Gm(t.R,e)&&(t.o[e]==1||t.o[e]==2)&&(t.o[e]=3,n.Mn(t.R,e,386)),t.o[e]==3&&!t.p[e]&&n.im(t.R,e)!=386&&(n.Vn(t.R,n.Zl(t.R,0,e)),n.Vn(t.R,n.Zl(t.R,1,e)))),n.im(t.R,e)==1&&t.o[e]==3&&!n.wm(t.R,n.Zl(t.R,0,e))&&!n.wm(t.R,n.Zl(t.R,1,e))&&(n.Vn(t.R,n.Zl(t.R,0,e)),n.Vn(t.R,n.Zl(t.R,1,e))),(n.cm(t.R,e)==1||n.cm(t.R,e)==2)&&(n.im(t.R,e)!=1||t.o[e]!=1&&t.o[e]!=2)&&(n.Vn(t.R,n.Zl(t.R,0,e)),n.Vn(t.R,n.Zl(t.R,1,e)));},n.ig=function(t){var A,e,r,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O;if(t.R.f!=0&&!t.G){for(t.I=0,O=0,e=1;e<t.R.f;e++)t.f[e]>t.f[O]&&(O=e);for(r=B(Q.xfb,Tt,6,t.R.f,16,1),D=B(Q.xfb,Tt,6,t.R.g,16,1),t.H=B(Q.S7,H,6,t.R.f,15,1),t.B=B(Q.S7,H,6,t.R.f,15,1),t.F=B(Q.S7,H,6,t.R.f,15,1),t.C=B(Q.S7,H,6,t.R.g,15,1),t.B[0]=O,t.H[O]=0,r[O]=true,o=0,l=1,U=0;o<t.R.f;)if(o<l){for(;;){for(b=0,g=0,T=-1,A=t.B[o],k=0;k<n.Mo(t.R,A);k++)(k<n.$o(t.R,A)||k>=n.Lo(t.R,A))&&(u=n.Zo(t.R,A,k),!r[u]&&t.f[u]>T&&(b=u,g=n._o(t.R,A,k),T=t.f[u]));if(T==-1)break;t.H[b]=l,t.F[l]=o,t.B[l++]=b,t.C[U++]=g,r[b]=true,D[g]=true;}++o;}else {for(c=0,T=-1,A=0;A<t.R.f;A++)!r[A]&&t.f[A]>T&&(c=A,T=t.f[A]);t.H[c]=l,t.F[l]=-1,t.B[l++]=c,r[c]=true;}for(t.D=B(Q.S7,H,6,2*(t.R.g-U),15,1);;){for(S=t.R.M,R=t.R.M,E=-1,i=0;i<t.R.g;i++)D[i]||(t.H[n.Zl(t.R,0,i)]<t.H[n.Zl(t.R,1,i)]?(F=t.H[n.Zl(t.R,0,i)],y=t.H[n.Zl(t.R,1,i)]):(F=t.H[n.Zl(t.R,1,i)],y=t.H[n.Zl(t.R,0,i)]),(F<S||F==S&&y<R)&&(S=F,R=y,E=i));if(E==-1)break;D[E]=true,t.C[U++]=E,t.D[2*t.I]=S,t.D[2*t.I+1]=R,++t.I;}t.G=true;}},n.jg=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b;for(o=null,b=n.Ep(t.R),y=0;y<b.j.a.length;y++)if(b.f[y]){for(i=0,T=s.xj(b.j,y),e=T,r=0,P=e.length;r<P;++r)A=e[r],n.yg(t,A)&&++i;if(i!=0)if(c=s.xj(b.k,y),o==null&&(o=B(Q.xfb,Tt,6,t.R.g,16,1)),i==T.length){for(l=-1,U=Ot,D=0;D<T.length;D++)U>t.B[c[D]]&&(U=t.B[c[D]],l=D);for(;i>0;)o[c[l]]=true,l=n.Mg(l+2,T.length),i-=2;}else {for(u=0;n.yg(t,T[u]);)++u;for(;!n.yg(t,T[u]);)u=n.Mg(u+1,T.length);for(;i>0;)for(o[c[u]]=true,u=n.Mg(u+2,T.length),i-=2;!n.yg(t,T[u]);)u=n.Mg(u+1,T.length);}}return o;},n.kg=function(t,A){var e,r,P,i,D,u,o,l,U;for(n.ig(t),l=new n.uw(t.R.f,t.R.g),n.On(l,t.R.K),D=0;D<t.R.f;D++)n.ll(t.R,l,t.B[D],0,0),n.on(l,D,t.bb[t.B[D]],t.ab[t.B[D]]);for(u=0;u<t.R.g;u++)n.nl(t.R,l,t.C[u],0,0,t.H,false),!(l.H[u]==257||l.H[u]==129)&&l.D[0][u]>l.D[1][u]&&(U=l.D[0][u],n.Dn(l,0,u,l.D[1][u]),l.D[1][u]=U,l.T=0),n.Gn(l,u,t.n[t.C[u]],t.k[t.C[u]]);if(A)for(i=0;i<t.R.f;i++)for(e=t.B[i],o=n.$o(t.R,e);o<n.Lo(t.R,e);o++)P=n.ll(t.R,l,n.Zo(t.R,e,o),0,0),n.ml(t.R,l,n._o(t.R,e,o),0,0,t.H[e],P,false);for(r=0;r<l.r;r++)e=l.D[0][r],t.eb[t.B[e]]&&(l.H[r]==257?(l.H[r]=129,l.T=0):l.H[r]==129&&(l.H[r]=257,l.T=0));return n.aw(t.R,l),l.T&=-9,l;},n.lg=function(t,A){return t.o[A];},n.mg=function(t){return n.ng(t,t.jb);},n.ng=function(t,A){return t.s==null&&(n.ig(t),n.cg(t,A,t.R.J)),t.s;},n.og=function(t){return t.N==null&&(n.ig(t),n.fg(t)),t.N;},n.pg=function(t){return n.ig(t),t.B;},n.qg=function(t){return n.ig(t),t.H;},n.rg=function(t){return t.J==null&&(n.ig(t),(t.Q&B1)==0&&(n.Ag(t),n.Bg(t,1),n.Bg(t,2)),n.zg(t)),t.J;},n.sg=function(t,A,e,r,P){var i,D,u;return i=e==-1?d.Math.abs(P[A].a-P[t.B[0]].a)/8:d.Math.abs(P[A].a-P[e].a),r<i&&(r=i),D=e==-1?d.Math.abs(P[A].b-P[t.B[0]].b)/8:d.Math.abs(P[A].b-P[e].b),r<D&&(r=D),t.jb&&(u=e==-1?d.Math.abs(P[A].c-P[t.B[0]].c)/8:d.Math.abs(P[A].c-P[e].c),r<u&&(r=u)),r;},n.tg=function(t,A){return t.X[A];},n.ug=function(t,A){return t.Y[A];},n.vg=function(t,A){return t.g==null?-1:t.g[A];},n.wg=function(t,A){return t.db[A];},n.xg=function(t,A){var e,r,P,i,D,u,o;for(o=n.Ep(t.R),e=0;e<o.j.a.length;e++)if(o.e[e]&&n.qt(o,e,A)){for(P=s.xj(o.j,e),i=0,D=P.length;i<D;++i)if(r=P[i],r!=A){for(u=0;u<n.$o(t.R,r);u++)if(n.Vp(t.R,n._o(t.R,r,u)))return  true;}return  false;}return  false;},n.yg=function(t,A){var e,r,P;if(n.Qo(t.R,A)<2)return  false;if(n.$o(t.R,A)==2)return  true;for(e=0,P=0;P<n.$o(t.R,A);P++)r=n._o(t.R,A,P),n.Up(t.R,r)&&(e+=n.fm(t.R,r)-1);return e>1;},n.zg=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt,Gt,dt,Bt,b1,f1,St,gt;if(n.bg(t,false),n._f(t,9,4),Bt=d.Math.max(n.Pg(t.R.f),n.Pg(t.R.g)),n._f(t,Q.Mfb(Bt),4),Bt==0){n._f(t,Mt(t.R.K?1:0),1),n._f(t,0,1),t.J=n.ag(t);return;}for(b1=St=f1=it=0,o=0;o<t.R.f;o++)if(S1(z(n.Ol(t.R,o),1),0)){switch(n.Tl(t.R,o)){case 6:break;case 7:++b1;break;case 8:++St;break;default:++f1;}n.Bl(t.R,o)!=0&&++it;}for(n._f(t,Mt(t.R.f),Bt),n._f(t,Mt(t.R.g),Bt),n._f(t,Q.Mfb(b1),Bt),n._f(t,Q.Mfb(St),Bt),n._f(t,Q.Mfb(f1),Bt),n._f(t,Q.Mfb(it),Bt),l=0;l<t.R.f;l++)n.Tl(t.R,t.B[l])==7&&S1(z(n.Ol(t.R,t.B[l]),1),0)&&n._f(t,Q.Mfb(l),Bt);for(T=0;T<t.R.f;T++)n.Tl(t.R,t.B[T])==8&&S1(z(n.Ol(t.R,t.B[T]),1),0)&&n._f(t,Q.Mfb(T),Bt);for(c=0;c<t.R.f;c++)n.Tl(t.R,t.B[c])!=6&&n.Tl(t.R,t.B[c])!=7&&n.Tl(t.R,t.B[c])!=8&&S1(z(n.Ol(t.R,t.B[c]),1),0)&&(n._f(t,Q.Mfb(c),Bt),n._f(t,Mt(n.Tl(t.R,t.B[c])),8));for(b=0;b<t.R.f;b++)n.Bl(t.R,t.B[b])!=0&&S1(z(n.Ol(t.R,t.B[b]),1),0)&&(n._f(t,Q.Mfb(b),Bt),n._f(t,Mt(8+n.Bl(t.R,t.B[b])),4));for(dt=0,O=0,g=1;g<t.R.f;g++)t.F[g]==-1?st=0:(st=1+t.F[g]-O,O=t.F[g]),dt<st&&(dt=st);for(bt=n.Pg(dt),n._f(t,Q.Mfb(bt),4),O=0,k=1;k<t.R.f;k++)t.F[k]==-1?st=0:(st=1+t.F[k]-O,O=t.F[k]),n._f(t,Q.Mfb(st),bt);for(Vt=0;Vt<2*t.I;Vt++)n._f(t,Mt(t.D[Vt]),Bt);for(G=0;G<t.R.g;G++)K=(n.hm(t.R,t.C[G])&LA)!=0||n.im(t.R,t.C[G])==16?1:n.Yp(t.R,t.C[G])?0:d.Math.min(3,n.fm(t.R,t.C[G])),n._f(t,Q.Mfb(K),2);if(e=0,(t.Q&B1)==0)for(F=0;F<t.R.f;F++)t._[t.B[F]]!=0&&t._[t.B[F]]!=3&&++e;if(n._f(t,Q.Mfb(e),Bt),(t.Q&B1)==0)for(F=0;F<t.R.f;F++)t._[t.B[F]]!=0&&t._[t.B[F]]!=3&&(n._f(t,Q.Mfb(F),Bt),t.bb[t.B[F]]==0?n._f(t,Mt(t._[t.B[F]]),3):(gt=t._[t.B[F]]==1?t.bb[t.B[F]]==1?4:6:t.bb[t.B[F]]==1?5:7,n._f(t,Q.Mfb(gt),3),n._f(t,Mt(t.ab[t.B[F]]),3)));if(A=0,(t.Q&B1)==0)for(J=0;J<t.R.g;J++)t.j[t.C[J]]!=0&&t.j[t.C[J]]!=3&&(!n.jq(t.R,t.C[J])||n.im(t.R,t.C[J])==1)&&++A;if(n._f(t,Q.Mfb(A),Bt),(t.Q&B1)==0)for(J=0;J<t.R.g;J++)t.j[t.C[J]]!=0&&t.j[t.C[J]]!=3&&(!n.jq(t.R,t.C[J])||n.im(t.R,t.C[J])==1)&&(n._f(t,Q.Mfb(J),Bt),n.im(t.R,t.C[J])==1?t.n[t.C[J]]==0?n._f(t,Mt(t.j[t.C[J]]),3):(gt=t.j[t.C[J]]==1?t.n[t.C[J]]==1?4:6:t.n[t.C[J]]==1?5:7,n._f(t,Q.Mfb(gt),3),n._f(t,Mt(t.k[t.C[J]]),3)):n._f(t,Mt(t.j[t.C[J]]),2));for(n._f(t,Mt(t.R.K?1:0),1),Dt=0,S=0;S<t.R.f;S++)n.Ml(t.R,t.B[S])!=0&&++Dt;if(Dt!=0)for(n.eg(t,1),n._f(t,Q.Mfb(Dt),Bt),F=0;F<t.R.f;F++)n.Ml(t.R,t.B[F])!=0&&(n._f(t,Q.Mfb(F),Bt),n._f(t,Mt(n.Ml(t.R,t.B[F])),8));if(t.w=0,t.R.K){for(n.gf(t,0,Bt,B1,1,-1),n.gf(t,3,Bt,Z1,1,-1),n.gf(t,4,Bt,120,4,3),n.gf(t,5,Bt,Xe,2,1),n.gf(t,6,Bt,1,1,-1),n.gf(t,7,Bt,X0,4,7),Dt=0,F=0;F<t.R.f;F++)n.Jl(t.R,t.B[F])!=null&&++Dt;if(Dt>0){for(n.eg(t,8),n._f(t,Q.Mfb(Dt),Bt),R=0;R<t.R.f;R++)if(E=n.Jl(t.R,t.B[R]),E!=null)for(n._f(t,Q.Mfb(R),Bt),n._f(t,Mt(E.length),4),P=E,i=0,D=P.length;i<D;++i)r=P[i],n._f(t,Q.Mfb(r),8);}n.hf(t,9,Bt,384,2,7),n.hf(t,10,Bt,31,5,0),n.gf(t,11,Bt,U0,1,-1),n.hf(t,12,Bt,LA,8,9),n.gf(t,13,Bt,$A,3,14),n.gf(t,14,Bt,aA,5,17),n.gf(t,16,Bt,FA,3,22);}for(Dt=0,U=0;U<t.R.f;U++)t.a!=null&&t.a[t.B[U]]!=-1&&++Dt;if(Dt!=0)for(n.eg(t,17),n._f(t,Q.Mfb(Dt),Bt),F=0;F<t.R.f;F++)t.a!=null&&t.a[t.B[F]]!=-1&&(n._f(t,Q.Mfb(F),Bt),n._f(t,Mt(t.a[t.B[F]]),4));if((t.Q&8)!=0||(t.Q&f0)!=0){for(Dt=0,Gt=0,F=0;F<t.R.f;F++)Jt=n.El(t.R,t.B[F]),Jt!=null&&(++Dt,Gt=d.Math.max(Gt,w.UAb(Jt).length));if(Dt!=0){for(Nt=n.Pg(Gt),n.eg(t,18),n._f(t,Q.Mfb(Dt),Bt),n._f(t,Q.Mfb(Nt),4),R=0;R<t.R.f;R++)if(Qt=n.El(t.R,t.B[R]),Qt!=null)for(n._f(t,Q.Mfb(R),Bt),n._f(t,Mt(w.UAb(Qt).length),Nt),ct=0;ct<w.UAb(Qt).length;ct++)n._f(t,(w.PAb(ct,w.UAb(Qt).length),Mt(w.UAb(Qt).charCodeAt(ct))),7);}}for(t.R.K&&(n.gf(t,19,Bt,mA,3,25),n.hf(t,20,Bt,iA,3,17)),Dt=0,y=0;y<t.R.f;y++)n.Pl(t.R,t.B[y])!=0&&++Dt;if(Dt!=0)for(n.eg(t,21),n._f(t,Q.Mfb(Dt),Bt),u=0;u<t.R.f;u++)n.Pl(t.R,t.B[u])!=0&&(n._f(t,Q.Mfb(u),Bt),n._f(t,Mt(n.Pl(t.R,t.B[u])>>4),2));if(t.R.K&&(n.gf(t,22,Bt,NA,1,-1),n.hf(t,23,Bt,TA,1,-1),n.hf(t,24,Bt,oe,2,21)),(t.Q&16)!=0){for(u=0;u<t.R.f;u++)if(n.Pm(t.R,t.B[u])){for(n.eg(t,25),r=0;r<t.R.f;r++)n._f(t,Mt(n.Pm(t.R,t.B[r])?1:0),1);break;}}if(Et=n.jg(t),Et!=null){for(Dt=0,J=0;J<t.R.g;J++)Et[t.C[J]]&&++Dt;for(n.eg(t,26),n._f(t,Q.Mfb(Dt),Bt),V=0;V<t.R.g;V++)Et[t.C[V]]&&n._f(t,Q.Mfb(V),Bt);}for(t.R.K&&n.gf(t,27,Bt,i0,1,-1),Dt=0,At=0;At<t.R.g;At++)n.im(t.R,t.C[At])==16&&++Dt;if(Dt!=0)for(n.eg(t,28),n._f(t,Q.Mfb(Dt),Bt),J=0;J<t.R.g;J++)n.im(t.R,t.C[J])==16&&n._f(t,Q.Mfb(J),Bt);for(t.R.K&&(n.gf(t,29,Bt,hA,2,30),n.gf(t,30,Bt,EA,7,32),n.gf(t,32,Bt,DP,2,44),n.gf(t,33,Bt,Un,5,39),n.gf(t,34,Bt,ee,1,-1),n.hf(t,35,Bt,Ln,1,-1),n.hf(t,36,Bt,96,2,5)),Dt=0,ot=0;ot<t.R.g;ot++)(n.im(t.R,t.C[ot])==32||n.im(t.R,t.C[ot])==64)&&++Dt;if(Dt!=0)for(n.eg(t,37),n._f(t,Q.Mfb(Dt),Bt),x=0;x<t.R.g;x++)(n.im(t.R,t.C[x])==32||n.im(t.R,t.C[x])==64)&&(n._f(t,Q.Mfb(x),Bt),n._f(t,Mt(n.im(t.R,t.C[x])==32?0:1),1));if(t.b&&(t.Q&16)!=0){for(Dt=0,lt=0,F=0;F<t.R.f;F++){for(ut=0,Ht=n.$o(t.R,t.B[F]);Ht<n.Lo(t.R,t.B[F]);Ht++)n.Pm(t.R,n.Zo(t.R,t.B[F],Ht))&&(wt=Ht-n.$o(t.R,t.B[F]),ut|=1<<wt,lt=d.Math.max(lt,wt+1));ut!=0&&++Dt;}if(Dt!=0)for(n.eg(t,38),n._f(t,Q.Mfb(Dt),Bt),n._f(t,Q.Mfb(lt),3),u=0;u<t.R.f;u++){for(ut=0,Ht=n.$o(t.R,t.B[u]);Ht<n.Lo(t.R,t.B[u]);Ht++)n.Pm(t.R,n.Zo(t.R,t.B[u],Ht))&&(wt=Ht-n.$o(t.R,t.B[u]),ut|=1<<wt);ut!=0&&(n._f(t,Q.Mfb(u),Bt),n._f(t,Q.Mfb(ut),lt));}}t.R.K&&n.gf(t,39,Bt,uP,4,47),n._f(t,0,1),t.J=n.ag(t);},n.Ag=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(t._=B(Q.O7,Y,6,t.R.f,15,1),A=0;A<t.R.f;A++)if(t.db[A]==1||t.db[A]==2){if(o=t.eb[A],n.Wp(t.R,A))for(u=0;u<n.$o(t.R,A);u++){for(P=n.Zo(t.R,A,u),y=0,U=B(Q.S7,H,6,3,15,1),l=0;l<n.$o(t.R,P);l++)U[y]=n.Zo(t.R,P,l),U[y]!=A&&++y;y==2&&t.f[U[0]]>t.f[U[1]]^t.H[U[0]]<t.H[U[1]]&&(o=!o);}else for(u=1;u<n.$o(t.R,A);u++)for(l=0;l<u;l++)i=n.Zo(t.R,A,u),D=n.Zo(t.R,A,l),t.f[i]>t.f[D]&&(o=!o),t.H[i]<t.H[D]&&(o=!o);t._[A]=t.db[A]==1^o?1:2;}else t._[A]=t.db[A];for(t.j=B(Q.O7,Y,6,t.R.g,15,1),e=0;e<t.R.g;e++)if(t.o[e]==1||t.o[e]==2){for(o=false,u=0;u<2;u++)if(r=n.Zl(t.R,u,e),n.$o(t.R,r)==3){for(U=B(Q.S7,H,6,2,15,1),y=0,l=0;l<3;l++)n.Zo(t.R,r,l)!=n.Zl(t.R,1-u,e)&&(U[y++]=n.Zo(t.R,r,l));t.f[U[0]]>t.f[U[1]]&&(o=!o),t.H[U[0]]<t.H[U[1]]&&(o=!o);}t.j[e]=t.o[e]==1^o?1:2;}else t.j[e]=t.o[e];},n.Bg=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b;for(o=B(Q.S7,H,6,32,15,1),l=0,r=0;r<t.R.f;r++)(t._[r]==1||t._[r]==2)&&t.bb[r]==A&&(u=t.ab[r],o[u]<t.f[r]&&(o[u]==0&&++l,o[u]=t.f[r]));for(i=0;i<t.R.g;i++)(t.j[i]==1||t.j[i]==2)&&t.n[i]==A&&n.im(t.R,i)==1&&(u=t.k[i],b=d.Math.max(t.f[n.Zl(t.R,0,i)],t.f[n.Zl(t.R,1,i)]),o[u]<b&&(o[u]==0&&++l,o[u]=b));for(D=B(Q.O7,Y,6,32,15,1),U=0;U<l;U++){for(T=-1,c=0,y=0;y<32;y++)c<o[y]&&(c=o[y],T=y);o[T]=0,D[T]=U<<24>>24;}for(e=0;e<t.R.f;e++)(t._[e]==1||t._[e]==2)&&t.bb[e]==A&&(t.ab[e]=D[t.ab[e]]);for(P=0;P<t.R.g;P++)(t.j[P]==1||t.j[P]==2)&&t.n[P]==A&&n.im(t.R,P)==1&&(t.k[P]=D[t.k[P]]);},n.Cg=function(t,A,e){var r,P;for(t.U=A,r=0;r<t.R.f;r++)t.f[r]=e[r],t.db[r]=0,t.ib[r]=false;for(P=0;P<t.R.g;P++)t.o[P]=0,t.q[P]=false;},n.Dg=function(t,A){var e,r,P;if(e=n.Zl(t.R,0,A),e>=t.R.f)return  false;if(t.db[e]==1||t.db[e]==2)return  true;if(t.db[e]==3)return  false;if(r=n.Ho(t.R,e),r!=-1)return t.o[r]==1||t.o[r]==2;for(P=0;P<n.$o(t.R,e);P++)if(n.ap(t.R,e,P)==2&&(t.db[n.Zo(t.R,e,P)]==1||t.db[n.Zo(t.R,e,P)]==2))return  true;return  false;},n.Eg=function(t,A){var e;for(e=0;e<n.$o(t.R,A);e++)if(t.o[n._o(t.R,A,e)]==3&&n.ap(t.R,A,e)==1)return  true;return  false;},n.Fg=function(t){var A,e,r,P,i,D,u;for(D=B(Q.S7,H,6,t.U+1,15,1),e=0;e<t.R.f;e++)n.Hl(t.R,e)==0&&(t.db[e]==1?++D[t.f[e]]:t.db[e]==2&&--D[t.f[e]]);for(P=0;P<t.R.g;P++)n.fm(t.R,P)==1&&n.cm(t.R,P)==0&&(t.o[P]==1?(++D[t.f[n.Zl(t.R,0,P)]],++D[t.f[n.Zl(t.R,1,P)]]):t.o[P]==2&&(--D[t.f[n.Zl(t.R,0,P)]],--D[t.f[n.Zl(t.R,1,P)]]));for(u=1;u<=t.U;u++)if(D[u]!=0){if(i=D[u]<0,i){for(A=0;A<t.R.f;A++)n.Hl(t.R,A)==0&&(t.db[A]==1?t.db[A]=2:t.db[A]==2&&(t.db[A]=1));for(r=0;r<t.R.g;r++)n.fm(t.R,r)==1&&n.cm(t.R,r)==0&&(t.o[r]==1?t.o[r]=2:t.o[r]==2&&(t.o[r]=1));}return i;}return  false;},n.Gg=function(t,A){var e;return e=B(Q.S7,H,6,A,15,1),h.Skb(t,0,e,0,t.length),e;},n.Hg=function(t,A){var e;return e=B(Q.xfb,Tt,6,A,16,1),h.Skb(t,0,e,0,t.length),e;},n.Ig=function(t){var A,e;if(t.$!=null)for(A=0;A<t.R.f;A++)n.hn(t.R,A,t.$[A]);if(t.i!=null)for(e=0;e<t.R.g;e++)n.Fn(t.R,e,t.i[e]);},n.Jg=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(A=0;A<t.R.f;A++)if(t.db[A]==1||t.db[A]==2){if(o=false,n.Wp(t.R,A))for(u=0;u<n.$o(t.R,A);u++){for(P=n.Zo(t.R,A,u),y=0,U=B(Q.S7,H,6,3,15,1),l=0;l<n.$o(t.R,P);l++)U[y]=n.Zo(t.R,P,l),U[y]!=A&&++y;y==2&&t.f[U[0]]>t.f[U[1]]^U[0]<U[1]&&(o=!o);}else for(u=1;u<n.$o(t.R,A);u++)for(l=0;l<u;l++)i=n.Zo(t.R,A,u),D=n.Zo(t.R,A,l),t.f[i]>t.f[D]&&(o=!o),i<D&&(o=!o);n.un(t.R,A,t.db[A]==1^o?1:2,t.fb[A]);}else n.un(t.R,A,t.db[A],t.fb[A]);for(e=0;e<t.R.g;e++)if(t.o[e]==1||t.o[e]==2){for(o=false,u=0;u<2;u++)if(r=n.Zl(t.R,u,e),n.$o(t.R,r)==3){for(U=B(Q.S7,H,6,2,15,1),y=0,l=0;l<3;l++)n.Zo(t.R,r,l)!=n.Zl(t.R,1-u,e)&&(U[y++]=n.Zo(t.R,r,l));t.f[U[0]]>t.f[U[1]]&&(o=!o),U[0]<U[1]&&(o=!o);}n.Jn(t.R,e,t.o[e]==1^o?1:2,t.p[e]);}else n.Jn(t.R,e,t.o[e],t.p[e]);},n.Kg=function(t){var A;for(A=0;A<t.R.f;A++)n.yn(t.R,A,t.M[A]);},n.Lg=function(t){var A,e,r;for(A=0;A<t.R.f;A++)!n.wm(t.R,A)&&t.db[A]==3&&n.ln(t.R,A,true);for(e=0;e<t.R.g;e++)t.o[e]==3&&(r=n.fm(t.R,e),r==1?n.ln(t.R,n.Zl(t.R,0,e),true):r==2&&n.Mn(t.R,e,386));},n.Mg=function(t,A){return t<A?t:t-A;},n.Ng=function(t){n.Og.call(this,t,0);},n.Og=function(t,A){this.R=t,this.Q=A,n.bw(this.R,7),this.c=n.Pg(this.R.f),(this.Q&B1)==0&&n.Cf(this),this.jb=(A&64)!=0||n.vm(this.R),this.b=n.np(this.R)==0,(this.Q&B1)==0&&(this.db=B(Q.O7,Y,6,this.R.f,15,1),this.fb=B(Q.xfb,Tt,6,this.R.f,16,1),this.ib=B(Q.xfb,Tt,6,this.R.f,16,1),this.o=B(Q.O7,Y,6,this.R.g,15,1),this.q=B(Q.xfb,Tt,6,this.R.g,16,1),this.p=B(Q.xfb,Tt,6,this.R.g,16,1)),this.d=false,n.Gf(this),(this.Q&B1)==0&&n.Mf(this),n.Lf(this);},n.Pg=function(t){var A;for(A=0;t>0;)t>>=1,++A;return A;},I(34,1,{},n.Ng,n.Og),L.b=false,L.c=0,L.d=false,L.r=false,L.t=0,L.v=0,L.w=0,L.G=false,L.I=0,L.K=false,L.L=false,L.O=0,L.Q=0,L.T=0,L.U=0,L.Z=false,L.jb=false,Q.i8=N(34),n.Qg=function(t,A){var e,r,P,i;if(t.d!=A.d)return t.d>A.d?1:-1;for(P=t.a.length,i=A.a.length,e=d.Math.min(P,i),r=0;r<e;r++)if(--P,--i,t.a[P]!=A.a[i])return t.a[P]>A.a[i]?1:-1;return P!=i?P>i?1:-1:t.b!=A.b?t.b>A.b?1:-1:0;},n.Rg=function(t,A){return n.Qg(t,A);},n.Sg=function(){},I(266,1,{},n.Sg),L.Rb=function(t,A){return n.Rg(t,A);},L.mb=function(t){return this===t;},Q._7=N(266),n.Vg=function(){},I(174,1,{174:1},n.Vg),L.b=0,L.c=0,L.d=0,Q.$7=N(174),n.Wg=function(t,A){return t.c!=A.c?t.c>A.c?1:-1:0;},n.Xg=function(t,A){return n.Wg(t,A);},n.Yg=function(){},I(267,1,{},n.Yg),L.Rb=function(t,A){return n.Xg(t,A);},L.mb=function(t){return this===t;},Q.b8=N(267),n._g=function(){},I(175,1,{175:1},n._g),L.a=0,L.b=0,L.c=0,Q.a8=N(175),n.ah=function(t,A,e){t.b==0&&(++t.c,t.b=63),t.b==63?(t.d[t.c]=Pt(t.d[t.c],e),t.b-=A):t.b>=A?(t.d[t.c]=c1(t.d[t.c],A),t.d[t.c]=Pt(t.d[t.c],e),t.b-=A):(t.d[t.c]=c1(t.d[t.c],t.b),t.d[t.c]=Pt(t.d[t.c],A0(e,A-t.b)),A-=t.b,++t.c,t.b=63-A,t.d[t.c]=Pt(t.d[t.c],z(e,Q.Mfb((1<<A)-1))));},n.bh=function(t,A){t.d[t.c]=v1(t.d[t.c],A);},n.dh=function(t,A){var e;for(e=0;e<t.c;e++)if(Rt(t.d[e],A.d[e]))return kn(t.d[e],A.d[e])?-1:1;return S1(t.d[t.c],A.d[t.c])?0:kn(t.d[t.c],A.d[t.c])?-1:1;},n.eh=function(t,A){t.a=A,t.c=0,t.b=63,s.Grb(t.d);},n.fh=function(t){this.d=B(Q.T7,e0,6,t,14,1);},I(101,1,{101:1,32:1},n.fh),L.Sb=function(t){return n.dh(this,t);},L.a=0,L.b=0,L.c=0,Q.c8=N(101),n.hh=function(t,A){return t.b!=A.b?t.b>A.b?-1:1:t.c!=A.c?t.c>A.c?-1:1:0;},n.ih=function(t,A,e){this.b=d.Math.max(t,A),this.c=d.Math.min(t,A),this.a=e;},I(173,1,{173:1,32:1},n.ih),L.Sb=function(t){return n.hh(this,t);},L.a=0,L.b=0,L.c=0,Q.d8=N(173),n.kh=function(t,A,e,r){this.a=(w.EAb(A),s.Drb(t,B(Q.S7,H,6,A,15,1),0,A)),this.b=(w.EAb(r),s.Drb(e,B(Q.S7,H,6,r,15,1),0,r));},I(208,1,{208:1},n.kh),Q.e8=N(208),n.lh=function(t,A){var e,r;for(r=0;r<t.g[A].length;r++)if(e=t.g[A][r],t.f[e]&&(t.o[e]==1||t.o[e]==2)&&t.k[e]==0)return  true;return  false;},n.mh=function(t,A,e){var r,P,i,D,u;for(P=0,D=0,u=0;u<t.g[A].length;u++)r=t.g[A][u],t.k[r]==e&&(i=1<<t.j[r],(D&i)==0&&(D|=i,++P));return P;},n.nh=function(t){var A,e,r,P,i,D,u,o,l,U,y,T;for(U=new s.Pxb(),y=0;y<t.i.f;y++)if(n.Qo(t.i,y)<2||n.$o(t.i,y)>2)for(D=1;D<n.$o(t.i,y);D++)for(A=n.Zo(t.i,y,D),u=0;u<D;u++)e=n.Zo(t.i,y,u),n.sh(t,A,e)&&(o=n.zh(t,A,e),o&&n.Gr(o,t.f)&&s.Rwb(U.a,o,(h.Ihb(),ve))==null);for(T=0;T<t.i.g;T++)t.c[T]!=0&&(n.fm(t.i,T)!=2||t.c[T]!=2)||(A=n.Zl(t.i,0,T),e=n.Zl(t.i,1,T),n.sh(t,A,e)&&(o=n.zh(t,A,e),o&&n.Gr(o,t.f)&&s.Rwb(U.a,o,(h.Ihb(),ve))==null));for(t.g=B(Q.S7,E1,7,U.a.c,0,2),t.e=B(Q.xfb,Tt,6,t.i.f,16,1),i=0,l=(P=new s.hxb(new s.nxb(new s.uqb(U.a).a).b),new s.Dqb(P));s.cpb(l.a.a);)for(o=(r=s.fxb(l.a),r.hd()),t.g[i++]=o.b,D=0;D<o.b.length;D++)t.e[o.b[D]]=true;},n.oh=function(t,A,e,r){var P,i,D,u,o,l,U;for(P=B(Q.S7,H,6,n.$o(t.i,e),15,1),u=0,D=0;D<n.$o(t.i,e);D++)P[u]=n.Zo(t.i,e,D),!r[P[u]]&&n.sh(t,A,P[u])&&++u;if(u==0)return  -1;if(u==1)return P[0];for(o=-1,l=Ot,i=0;i<u;i++)U=n.Bp(t.i,A,P[i],Ot,r),U<l&&(l=U,o=P[i]);return o;},n.ph=function(t,A){return t.f[A]&&(t.o[A]==1||t.o[A]==2);},n.qh=function(t,A){return t.e[A];},n.rh=function(t){var A,e;for(e=true,A=0;A<t.i.f;A++)if(t.o[A]!=0&&!t.e[A]){e=false;break;}return e;},n.sh=function(t,A,e){var r,P,i,D,u;if(A==e||t.a[A]!=t.a[e]||t.o[A]!=0&&(t.o[A]==3||t.o[e]==3||t.p[A]^t.o[A]!==t.o[e]||t.k[A]!==t.k[e]||t.j[A]!==t.j[e])||(r=n.Xo(t.i,A,e),r!=-1&&(n.fm(t.i,r)==1&&t.c[r]!=0||n.fm(t.i,r)==2&&t.c[r]==1)))return  false;if(n.Qo(t.i,A)==1&&!n.Tp(t.i,A)){for(P=-1,u=0;u<n.$o(t.i,A);u++)if(n.Zo(t.i,A,u)!=e&&n.ap(t.i,A,u)==2){P=n._o(t.i,A,u);break;}for(i=-1,D=0;D<n.$o(t.i,e);D++)if(n.Zo(t.i,e,D)!=A&&n.ap(t.i,e,D)==2){i=n._o(t.i,e,D);break;}if(P!=-1&&t.c[P]!=0&&t.d[P]^t.c[P]===t.c[i])return  false;}return  true;},n.th=function(t,A,e,r,P){var i,D,u,o,l,U;for(o=null,i=null,U=0;U<t.g[A].length;U++)D=t.g[A][U],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?i=n.Bh(i,(P[D]<<16)+D):t.k[D]==r&&t.j[D]==e&&(o=n.Bh(o,(P[D]<<16)+D)));if(u=n.Ph(o,i),u==0)return  false;if(u<0)for(l=0;l<t.g[A].length;l++)D=t.g[A][l],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?(t.k[D]=r<<24>>24,t.j[D]=e<<24>>24):t.k[D]==r&&t.j[D]==e&&(t.k[D]=0,t.j[D]=-1));return  true;},n.uh=function(t,A){var e,r,P,i,D,u;if(!t.b)return  false;for(P=false,i=t.b.a.length-1;i>=0;i--)if(r=false,D=s.xj(t.b,i),D.a==2?r=n.th(t,D.b,D.c,D.d,A):D.a==1&&(r=n.yh(t,D.b,A)),r){for(s.Fj(t.b,D),u=0;u<t.g[D.b].length;u++)e=t.g[D.b][u],t.n[e]=false;P=true;}return P;},n.vh=function(t){var A,e,r,P,i,D,u,o;if(t.g!=null)for(D=new n.Oh(t),t.b=new s.Mj(),P=0;P<t.g.length;P++)r=n.Gh(D,P),r==0?(n.Ch(D,P),u=n.mh(t,P,2),A=n.mh(t,P,1),e=n.lh(t,P),u==1&&A==1&&!e&&(n.xh(t,P,D.a+D.f++),s.sj(t.b,new n.jk(P,1,-1,-1))),u>0?(e&&(n.wh(t,P,D.i+D.g++,2),++u),s.sj(t.b,new n.jk(P,1,-1,-1))):A>0?(e&&n.wh(t,P,D.a+D.f++,1),s.sj(t.b,new n.jk(P,1,-1,-1))):e&&(n.wh(t,P,D.a+D.f++,1),s.sj(t.b,new n.jk(P,1,-1,-1)))):r==1&&(n.lh(t,P)?(i=n.Fh(D,P),o=n.Hh(D,P),s.sj(t.b,new n.jk(P,2,i,o))):(n.Ch(D,P),s.sj(t.b,new n.jk(P,1,-1,-1))));},n.wh=function(t,A,e,r){var P,i;for(i=0;i<t.g[A].length;i++)P=t.g[A][i],t.f[P]&&(t.o[P]==1||t.o[P]==2)&&t.k[P]==0&&(t.k[P]=r<<24>>24,t.j[P]=e<<24>>24);},n.xh=function(t,A,e){var r,P;for(P=0;P<t.g[A].length;P++)r=t.g[A][P],t.k[r]==2&&(t.k[r]=1,t.j[r]=e<<24>>24);},n.yh=function(t,A,e){var r,P,i,D,u,o,l,U;for(i=t.g[A],P=1,o=0;o<i.length;o++)if(r=i[o],t.f[r]&&t.k[r]==2){P=2;break;}for(D=B(Q.S7,E1,7,32,0,2),l=0;l<i.length;l++)r=i[l],t.f[r]&&t.k[r]==P&&(D[t.j[r]]=n.Bh(D[t.j[r]],(e[r]<<16)+r));for(U=0;U<32;U++)D[U]!=null&&w.nAb(D[U],w.UAb(T0(s.nsb.prototype.ld,s.nsb,[])));if(s.Xrb(D,new n.Qh()),n.Ph(D[0],D[1])==0)return  false;for(u=0;u<D[0].length;u++)r=D[0][u]&n1,t.k[r]=0,t.j[r]=-1;return  true;},n.zh=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O;for(o=B(Q.S7,H,6,t.i.f,15,1),F=B(Q.S7,H,6,t.i.f,15,1),b=B(Q.xfb,Tt,6,t.i.f,16,1),l=B(Q.xfb,Tt,6,t.i.f,16,1),S=new n.Hr(t.i.f),o[0]=A,F[A]=e,F[e]=-2,S.a[A]=true,S.a[e]=true,i=0,U=0;i<=U;){if(D=o[i],F[D]==D){for(y=0;y<n.$o(t.i,D);y++)if(r=n.Zo(t.i,D,y),!S.a[r]){if(n.ap(t.i,D,y)==2&&n.Tl(t.i,r)<10)o[++U]=r,F[r]=r,l[r]=l[D]||n.Qo(t.i,r)==2,b[r]=l[D]&&!b[D],S.a[r]=true;else if(l[D]&&b[D]){if(R=n.oh(t,r,F[D],S.a),R==-1)return null;o[++U]=r,F[r]=R,F[R]=-2,l[r]=false,S.a[r]=true,S.a[R]=true;}else if(n.gq(t.i,n._o(t.i,D,y))&&(o[++U]=r,F[r]=r,l[r]=false,S.a[r]=true,(n.Tl(t.i,r)==6&&n.Qo(t.i,r)==0||n.Tl(t.i,r)==7&&n.Bl(t.i,r)==1||n.Tl(t.i,r)==14||n.Tl(t.i,r)==15&&n.$o(t.i,r)>2||n.Tl(t.i,r)==16&&n.$o(t.i,r)>2)&&n.$o(t.i,r)>2)){for(u=false,g=1;g<n.$o(t.i,r);g++)if(E=n.Zo(t.i,r,g),!S.a[E])for(k=0;k<g;k++)O=n.Zo(t.i,r,k),S.a[O]||n.sh(t,E,O)&&(o[++U]=E,F[E]=O,F[O]=-2,l[E]=false,S.a[E]=true,S.a[O]=true,u=true);if(!u)return null;}}}else {for(P=B(Q.xfb,Tt,6,n.$o(t.i,D),16,1),T=0;T<n.$o(t.i,D);T++)if(r=n.Zo(t.i,D,T),S.a[r])P[T]=F[r]==r;else for(g=0;g<n.$o(t.i,r);g++)if(n.Zo(t.i,r,g)==F[D]){P[T]=true;break;}for(c=0;c<n.$o(t.i,D);c++)if(P[c])if(r=n.Zo(t.i,D,c),S.a[r]){if(n.Xo(t.i,r,F[D])==-1)return null;}else o[++U]=r,F[r]=r,b[r]=false,l[r]=true,S.a[r]=true;for(y=0;y<n.$o(t.i,D);y++)if(!P[y]&&(r=n.Zo(t.i,D,y),!S.a[r])){if(R=n.oh(t,r,F[D],S.a),R==-1)return null;o[++U]=r,F[r]=R,F[R]=-2,l[r]=false,S.a[r]=true,S.a[R]=true;}}++i;}return S;},n.Ah=function(t,A,e,r,P,i,D,u,o,l){this.i=t,this.a=A,this.f=e,this.o=r,this.c=P,this.k=i,this.j=D,this.p=u,this.d=o,this.n=l,n.nh(this);},n.Bh=function(t,A){var e,r;for(r=B(Q.S7,H,6,t==null?1:t.length+1,15,1),e=0;e<r.length-1;e++)r[e]=t[e];return r[r.length-1]=A,r;},I(265,1,{},n.Ah),Q.g8=N(265),n.Ch=function(t,A){var e,r,P,i,D,u;for(P=0;P<t.b;P++)if(t.e[P][A]&&t.c[P]!=-3){for(r=0;r<=t.j.g.length;r++)if(r!=A&&t.e[P][r])for(t.e[P][A]=false,u=P<t.a?P:P<t.b?P-t.a:-1,D=n.Mh(t,P<t.a?1:P<t.b?2:0),i=0;i<t.j.g[A].length;i++)e=t.j.g[A][i],n.ph(t.j,e)&&t.j.j[e]==u&&(t.j.j[e]=D<<24>>24);}},n.Dh=function(t,A,e){var r,P,i,D,u;for(u=false,D=1,A[e]=D,r=true;r;){for(r=false,P=0;P<t.b;P++)if(A[P]==D)for(i=0;i<t.b;i++)A[i]==0&&n.Lh(t,P,i)&&(t.c[i]==-2?(A[i]=D+1,r=true):t.c[i]!=t.c[e]&&(A[i]=D+1,u=true));++D;}return u;},n.Eh=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b;for(o=r+1;o<t.j.g.length;o++)if(o!=r&&t.e[A][o]&&t.e[e][o])return D=B(Q.S7,H,6,2,15,1),D[0]=e,D[1]=A,D;for(b=B(Q.S7,H,6,t.b,15,1),U=B(Q.S7,H,6,t.b,15,1),l=B(Q.S7,H,6,t.b,15,1),i=0,y=0,l[0]=A,U[A]=1;i<=y;){for(T=0;T<t.d[l[i]].length;T++){if(P=t.d[l[i]][T],P==e){if(i==0)continue;for(u=U[l[i]]+1,D=B(Q.S7,H,6,u,15,1),D[0]=P,D[1]=l[i],c=2;c<u;c++)D[c]=b[D[c-1]];return D;}U[P]==0&&t.c[P]!=-3&&(U[P]=U[l[i]]+1,l[++y]=P,b[P]=l[i]);}++i;}return null;},n.Fh=function(t,A){var e;for(e=0;e<t.b;e++)if(t.e[e][A]&&t.c[e]==-3)return e<t.a?e:e<t.b?e-t.a:-1;return  -1;},n.Gh=function(t,A){var e,r;for(e=0,r=0;r<t.b;r++)t.e[r][A]&&t.c[r]==-3&&++e;return e;},n.Hh=function(t,A){var e;for(e=0;e<t.b;e++)if(t.e[e][A]&&t.c[e]==-3)return e<t.a?1:e<t.b?2:0;return  -1;},n.Ih=function(t,A){return A<t.a?A:A<t.b?A-t.a:-1;},n.Jh=function(t,A){return A<t.a?1:A<t.b?2:0;},n.Kh=function(t,A){var e,r;return r=t.j.k[A],e=t.j.j[A],r==0?t.b:r==1?e:t.a+e;},n.Lh=function(t,A,e){var r;for(r=0;r<t.j.g.length;r++)if(t.e[A][r]&&t.e[e][r])return  true;return  false;},n.Mh=function(t,A){return A==1?t.a+t.f++:t.i+t.g++;},n.Nh=function(t,A){var e,r,P,i,D,u,o,l,U,y;for(U=Ot,o=-1,y=-1,l=-1,r=0;r<t.j.i.f;r++)if(n.ph(t.j,r)&&t.j.k[r]!=0)for(u=0;u<A.length;u++)P=n.Ih(t,A[u]),i=n.Jh(t,A[u]),t.j.k[r]==i&&t.j.j[r]==P&&U>t.j.a[r]+(i==1?H1:0)&&(U=t.j.a[r]+(i==1?H1:0),o=P,y=i,l=A[u]);for(e=0;e<t.j.i.f;e++)n.ph(t.j,e)&&t.j.k[e]==y&&t.j.j[e]==o&&(t.j.k[e]=0,t.j.j[e]=-1);for(D=0;D<t.j.g.length;D++)t.e[l][D]=false;},n.Oh=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k;for(this.j=t,r=0;r<t.i.f;r++)t.f[r]&&(t.o[r]==1||t.o[r]==2)&&(t.k[r]==1?this.a<=t.j[r]&&(this.a=1+t.j[r]):t.k[r]==2&&this.i<=t.j[r]&&(this.i=1+t.j[r]));for(this.b=this.a+this.i,this.e=T1(Q.xfb,[Q1,Tt],[12,6],16,[this.b+1,t.g.length+1],2),P=0;P<t.i.f;P++)t.f[P]&&(t.o[P]==1||t.o[P]==2)&&!t.e[P]&&(this.e[n.Kh(this,P)][t.g.length]=true);for(o=0;o<t.g.length;o++)for(k=0;k<t.g[o].length;k++)e=t.g[o][k],t.f[e]&&(t.o[e]==1||t.o[e]==2)&&(this.e[n.Kh(this,e)][o]=true);for(this.d=B(Q.S7,E1,7,this.b,0,2),l=0;l<t.g.length;l++)for(c=1;c<this.b;c++)if(this.e[c][l])for(b=0;b<c;b++)this.e[b][l]&&(this.d[c]=n.Bh(this.d[c],b),this.d[b]=n.Bh(this.d[b],c));for(this.c=B(Q.S7,H,6,this.b+1,15,1),T=0;T<this.b;T++)this.e[T][t.g.length]?this.c[T]=-1:this.c[T]=-2;for(U=0;U<t.g.length;U++)if(this.e[this.b][U])for(y=0;y<this.b;y++)this.e[y][U]&&this.c[y]!=U&&(this.c[y]==-2?this.c[y]=U:this.c[y]=-3);for(A=0;A<this.b;A++)if(this.c[A]>=-1&&(i=B(Q.S7,H,6,this.b,15,1),n.Dh(this,i,A)))for(y=0;y<this.b;y++)i[y]!=0&&(this.c[y]=-3);for(u=0;u<t.g.length-1;u++)for(c=1;c<this.b;c++)if(this.e[c][u]&&this.c[c]!=-3){for(b=0;b<c;b++)if(this.e[b][u]&&this.c[b]!=-3&&(D=n.Eh(this,c,b,u),D!=null)){for(g=0;g<D.length;g++)this.c[D[g]]=-3;n.Nh(this,D);break;}}},I(366,1,{},n.Oh),L.a=0,L.b=0,L.f=0,L.g=0,L.i=0,Q.f8=N(366),n.Ph=function(t,A){var e,r;if(t==null)return A==null?0:1;if(A==null)return  -1;for(e=d.Math.min(t.length,A.length),r=0;r<e;r++)if((t[r]&fA)!=(A[r]&fA))return (t[r]&fA)<(A[r]&fA)?-1:1;return t.length==A.length?0:t.length<A.length?-1:1;},n.Qh=function(){},I(367,1,{},n.Qh),L.Rb=function(t,A){return n.Ph(t,A);},L.mb=function(t){return this===t;},Q.h8=N(367),n.Th=function(t,A){switch(A){case 0:return n.Uh(t,false);case 1:return n.Wh(t);case 2:return n.Vh(t);case 3:return n.Yh(t);case 4:return n.Xh(t);}return null;},n.Uh=function(t,A){try{return A?(t=n.fw(t),n.zq(t,!0),n.Tr(t),n.rg(new n.Ng(t))):n.rg(new n.Ng(t));}catch(e){if(e=M1(e),Zt(e,27))return h.Rkb(),null;throw Lt(e);}},n.Vh=function(t){var A,e;try{for(t=n.fw(t),n.pw(t),A=t.r,e=0;e<A;e++)t.H[e]=1,t.T=0;return n.rg(new n.Ng(t));}catch(r){if(r=M1(r),Zt(r,27))return h.Rkb(),null;throw Lt(r);}},n.Wh=function(t){try{return t=n.fw(t),n.pw(t),n.rg(new n.Ng(t));}catch(A){if(A=M1(A),Zt(A,27))return h.Rkb(),null;throw Lt(A);}},n.Xh=function(t){var A;try{return t=n.fw(t),n.pw(t),A=n.Hw(new n.Kw(t)),n.rg(new n.Og(A,8));}catch(e){if(e=M1(e),Zt(e,27))return h.Rkb(),null;throw Lt(e);}},n.Yh=function(t){var A;try{return A=n.Hw(new n.Kw(t)),n.rg(new n.Og(A,8));}catch(e){if(e=M1(e),Zt(e,27))return h.Rkb(),null;throw Lt(e);}},n.Zh=function(t,A,e,r){t.a+=A,t.b+=e,t.c+=r;},n.$h=function(t,A){return t.a+=A.a,t.b+=A.b,t.c+=A.c,t;},n._h=function(t,A){return new n.oi(t.a+A.a,t.b+A.b,t.c+A.c);},n.ai=function(t,A){return t.a=(t.a+A.a)/2,t.b=(t.b+A.b)/2,t.c=(t.c+A.c)/2,t;},n.bi=function(t,A){return t.a!=A.a?t.a<A.a?-1:1:t.b!=A.b?t.b<A.b?-1:1:t.c!=A.c?t.c<A.c?-1:1:0;},n.ci=function(t,A){return new n.oi(t.b*A.c-t.c*A.b,-(t.a*A.c-t.c*A.a),t.a*A.b-t.b*A.a);},n.di=function(t,A){return d.Math.sqrt((A.a-t.a)*(A.a-t.a)+(A.b-t.b)*(A.b-t.b)+(A.c-t.c)*(A.c-t.c));},n.ei=function(t,A){var e;return A==null||!Zt(A,29)?false:(e=A,d.Math.abs(e.a-t.a)+d.Math.abs(e.b-t.b)+d.Math.abs(e.c-t.c)<1e-6);},n.fi=function(t,A){var e,r,P;return r=t.a*t.a+t.b*t.b+t.c*t.c,P=A.a*A.a+A.b*A.b+A.c*A.c,r==0||P==0||(e=(t.a*A.a+t.b*A.b+t.c*A.c)/d.Math.sqrt(r*P),e>=1)?0:e<=-1?t1:d.Math.acos(e);},n.gi=function(t,A){var e,r,P;return e=t.a,r=t.b,P=t.c,t.a=e*A[0][0]+r*A[1][0]+P*A[2][0],t.b=e*A[0][1]+r*A[1][1]+P*A[2][1],t.c=e*A[0][2]+r*A[1][2]+P*A[2][2],t;},n.hi=function(t,A){return t.a*=A,t.b*=A,t.c*=A,t;},n.ii=function(t,A,e,r){t.a=A,t.b=e,t.c=r;},n.ji=function(t,A){return n.ii(t,A.a,A.b,A.c),t;},n.ki=function(t,A){return t.a-=A.a,t.b-=A.b,t.c-=A.c,t;},n.li=function(t,A){return new n.oi(t.a-A.a,t.b-A.b,t.c-A.c);},n.mi=function(t){var A;return A=d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),A==0?(h.Rkb(),t.a=1,t.b=0,t.c=0,t):(t.a/=A,t.b/=A,t.c/=A,t);},n.ni=function(){},n.oi=function(t,A,e){this.a=t,this.b=A,this.c=e;},n.pi=function(t){n.oi.call(this,t.a,t.b,t.c);},I(29,1,{29:1,4:1,32:1},n.ni,n.oi,n.pi),L.Sb=function(t){return n.bi(this,t);},L.mb=function(t){return n.ei(this,t);},L.pb=function(){var t;return t=new Q0.Jnb("0.00"),"["+Q0.Fnb(t,this.a)+", "+Q0.Fnb(t,this.b)+", "+Q0.Fnb(t,this.c)+"]";},L.a=0,L.b=0,L.c=0,Q.j8=N(29),n.ti=function(t,A){A.Nb(t.c),n.le(A,t.a,t.b);},n.ui=function(t,A){A.c*=t.c,A.a=A.a*t.c+t.a,A.b=A.b*t.c+t.b;},n.vi=function(t,A){n.dn(A,t.c),n._n(A,t.a,t.b);},n.wi=function(t,A){A.a=A.a*t.c+t.a,A.b=A.b*t.c+t.b;},n.xi=function(t,A){A.c=A.c*t.c+t.a,A.d=A.d*t.c+t.b,A.b*=t.c,A.a*=t.c;},n.yi=function(t){t.a=0,t.b=0,t.c=1;},n.zi=function(t){return t.c==1&&t.a==0&&t.b==0;},n.Ai=function(t,A){return A*t.c+t.a;},n.Bi=function(t,A){return A*t.c+t.b;},n.Ci=function(){n.yi(this);},n.Di=function(t,A){this.c=1,this.a=t,this.b=A;},n.Ei=function(t,A,e,r){var P,i,D,u;n.yi(this),A?(r&Ye)==0?ht.lO(A,t.c,t.d)&&ht.lO(A,t.c+t.b,t.d+t.a)||((t.b>A.b||t.a>A.a)&&(i=A.b/t.b,u=A.a/t.a,this.c=d.Math.min(i,u)),t.c*this.c<A.c?this.a=A.c-t.c*this.c:(t.c+t.b)*this.c>A.c+A.b&&(this.a=A.c+A.b-(t.c+t.b)*this.c),t.d*this.c<A.d?this.b=A.d-t.d*this.c:(t.d+t.a)*this.c>A.d+A.a&&(this.b=A.d+A.a-(t.d+t.a)*this.c)):(i=A.b/t.b,u=A.a/t.a,D=r&n1,D==0?D=24:(r&r0)!=0&&(D/=256),P=D/e,this.c=d.Math.min(P,d.Math.min(i,u)),this.a=A.c+A.b/2-this.c*(t.c+t.b/2),this.b=A.d+A.a/2-this.c*(t.d+t.a/2)):(r&H1)!=0&&(D=(r&n1)!=0?r&n1:24,this.c=D/e);},I(78,1,{},n.Ci,n.Di,n.Ei),L.pb=function(){return "DepictorTransformation Offset: "+this.a+","+this.b+" Scaling: "+this.c;},L.a=0,L.b=0,L.c=0,Q.k8=N(78),n.Gi=function(t){var A,e,r,P;return !h.ckb(w.UAb(t).substr(0,21),iP)||(r=w.UAb(Ze).length,!h.ckb(w.UAb(t).substr(w.UAb(t).length-r,r),Ze))||(e=h.kkb(t,X1(34),21),e==-1)?null:(P=(w.OAb(21,e,w.UAb(t).length),w.UAb(t).substr(21,e-21)),A=h.tkb(t,21+w.UAb(P).length+1,w.UAb(t).length-17),h.ckb(P,"arrow")?new tt.IH(A):h.ckb(P,"text")?new n.bx(A):null);},s.Asb=function(t){return new s.Cwb(t);},s.Bsb=function(t){return new Ft.Xzb(t.Ub());},s.Hi=function(t,A){var e,r,P;for(w.IAb(A),e=false,P=A.Tb();P.ed();)r=P.fd(),e=e|t.add(r);return e;},s.Ii=function(t,A,e){var r,P;for(P=t.Tb();P.ed();)if(r=P.fd(),j1(A)===j1(r)||A!=null&&h.Oc(A,r))return e&&P.gd(),true;return  false;},s.Ji=function(t,A){var e,r;for(w.IAb(A),r=A.Tb();r.ed();)if(e=r.fd(),!t.contains(e))return  false;return  true;},s.Ki=function(t,A){var e,r,P;for(w.IAb(A),e=false,r=t.Tb();r.ed();)P=r.fd(),A.contains(P)&&(r.gd(),e=true);return e;},s.Li=function(t,A){var e,r,P;for(w.IAb(A),e=false,r=t.Tb();r.ed();)P=r.fd(),A.contains(P)||(r.gd(),e=true);return e;},s.Mi=function(t,A){var e,r,P,i;for(i=t.size(),A.length<i&&(A=w.oAb(new Array(i),A)),P=A,r=t.Tb(),e=0;e<i;++e)a1(P,e,r.fd());return A.length>i&&Q.B6(A,i,null),A;},s.Ni=function(t){var A,e,r;for(r=new s.Hwb(", ","[","]"),e=t.Tb();e.ed();)A=e.fd(),s.Ewb(r,j1(A)===j1(t)?"(this Collection)":A==null?PA:Xn(A));return r.a?w.UAb(r.e).length==0?r.a.a:r.a.a+(""+r.e):r.c;},I(43,1,_e),L.Ub=function(){return new s.Cwb(this);},L.Vb=function(){return new Ft.Xzb(this.Ub());},L.add=function(t){throw Lt(new h.Vkb("Add not supported on this collection"));},L.addAll=function(t){return s.Hi(this,t);},L.clear=function(){var t;for(t=this.Tb();t.ed();)t.fd(),t.gd();},L.contains=function(t){return s.Ii(this,t,false);},L.containsAll=function(t){return s.Ji(this,t);},L.isEmpty=function(){return this.size()==0;},L.remove=function(t){return s.Ii(this,t,true);},L.removeAll=function(t){return s.Ki(this,t);},L.retainAll=function(t){return s.Li(this,t);},L.toArray=function(){return this.Wb(B(Q.Kcb,C1,1,this.size(),5,1));},L.Wb=function(t){return s.Mi(this,t);},L.pb=function(){return s.Ni(this);},Q.$cb=N(43),s.uvb=function(t){return new s.Cwb(t);},s.vvb=function(t){var A;return Ft.Vzb(t.Vb(),(A=Ft.dzb(new Ft.kzb(),Ft.Tyb(new Ft.izb(),new Ft.gzb(),new Ft.Kzb(),j(v(Q.qeb,1),y0,96,0,[(Ft.Xyb(),Ft.Vyb)]))),Ft._yb(A,new Ft.nzb())));},s.aj=function(t,A){var e,r;for(e=0,r=t.size();e<r;++e)if(s.Jvb(A,t.getAtIndex(e)))return e;return  -1;},s.bj=function(t,A,e){return new s.npb(t,A,e);},I(95,43,K8),L.Ub=function(){return new s.Cwb(this);},L.addAtIndex=function(t,A){throw Lt(new h.Vkb("Add not supported on this list"));},L.add=function(t){return this.addAtIndex(this.size(),t),true;},L.addAllAtIndex=function(t,A){var e,r,P;for(w.IAb(A),e=false,P=A.Tb();P.ed();)r=P.fd(),this.addAtIndex(t++,r),e=true;return e;},L.clear=function(){this.Xb(0,this.size());},L.mb=function(t){var A,e,r,P,i;if(t===this)return  true;if(!Zt(t,83)||(i=t,this.size()!=i.size()))return  false;for(P=i.Tb(),e=this.Tb();e.ed();)if(A=e.fd(),r=P.fd(),!(j1(A)===j1(r)||A!=null&&h.Oc(A,r)))return  false;return  true;},L.ob=function(){return s.Dsb(this);},L.indexOf=function(t){return s.aj(this,t);},L.Tb=function(){return new s.fpb(this);},L.lastIndexOf=function(t){var A;for(A=this.size()-1;A>-1;--A)if(s.Jvb(t,this.getAtIndex(A)))return A;return  -1;},L.removeAtIndex=function(t){throw Lt(new h.Vkb("Remove not supported on this list"));},L.Xb=function(t,A){var e,r;for(r=new s.lpb(this,t),e=t;e<A;++e)w.GAb(r.a<r.c.size()),r.c.getAtIndex(r.b=r.a++),s.epb(r);},L.setAtIndex=function(t,A){throw Lt(new h.Vkb("Set not supported on this list"));},L.subList=function(t,A){return s.bj(this,t,A);},Q.fdb=N(95);function _u(f){return f==null?false:f.$implements__java_lang_Cloneable||Array.isArray(f);}s.qj=function(t){t.a=B(Q.Kcb,C1,1,0,5,1);},s.rj=function(t,A,e){w.KAb(A,t.a.length),w.hAb(t.a,A,e);},s.sj=function(t,A){return w.kAb(t.a,A),true;},s.tj=function(t,A,e){var r,P;return w.KAb(A,t.a.length),r=e.toArray(),P=r.length,P==0?false:(w.iAb(t.a,A,r),true);},s.uj=function(t,A){var e,r;return e=A.toArray(),r=e.length,r==0?false:(w.iAb(t.a,t.a.length,e),true);},s.vj=function(t){w.mAb(t.a,0);},s.wj=function(t,A){return s.zj(t,A,0)!=-1;},s.xj=function(t,A){return w.HAb(A,t.a.length),t.a[A];},s.yj=function(t,A){return s.zj(t,A,0);},s.zj=function(t,A,e){for(;e<t.a.length;++e)if(s.Jvb(A,t.a[e]))return e;return  -1;},s.Aj=function(t){return t.a.length==0;},s.Bj=function(t){return new s.trb(t);},s.Cj=function(t,A){return s.Dj(t,A,t.a.length-1);},s.Dj=function(t,A,e){for(;e>=0;--e)if(s.Jvb(A,t.a[e]))return e;return  -1;},s.Ej=function(t,A){var e;return e=(w.HAb(A,t.a.length),t.a[A]),w.lAb(t.a,A,1),e;},s.Fj=function(t,A){var e;return e=s.zj(t,A,0),e==-1?false:(s.Ej(t,e),true);},s.Gj=function(t,A,e){var r;w.LAb(A,e,t.a.length),r=e-A,w.lAb(t.a,A,r);},s.Hj=function(t,A,e){var r;return r=(w.HAb(A,t.a.length),t.a[A]),t.a[A]=e,r;},s.Ij=function(t){return t.a.length;},s.Jj=function(t,A){s.Wrb(t.a,t.a.length,A);},s.Kj=function(t){return w.eAb(t.a);},s.Lj=function(t,A){var e,r;for(r=t.a.length,A.length<r&&(A=w.oAb(new Array(r),A)),e=0;e<r;++e)a1(A,e,t.a[e]);return A.length>r&&Q.B6(A,r,null),A;},s.Mj=function(){s.qj(this);},s.Nj=function(t){s.qj(this),w.CAb(t>=0,"Initial capacity must not be negative");},I(20,95,We,s.Mj,s.Nj),L.addAtIndex=function(t,A){s.rj(this,t,A);},L.add=function(t){return s.sj(this,t);},L.addAllAtIndex=function(t,A){return s.tj(this,t,A);},L.addAll=function(t){return s.uj(this,t);},L.clear=function(){s.vj(this);},L.contains=function(t){return s.wj(this,t);},L.getAtIndex=function(t){return s.xj(this,t);},L.indexOf=function(t){return s.yj(this,t);},L.isEmpty=function(){return s.Aj(this);},L.Tb=function(){return s.Bj(this);},L.lastIndexOf=function(t){return s.Cj(this,t);},L.removeAtIndex=function(t){return s.Ej(this,t);},L.remove=function(t){return s.Fj(this,t);},L.Xb=function(t,A){s.Gj(this,t,A);},L.setAtIndex=function(t,A){return s.Hj(this,t,A);},L.size=function(){return s.Ij(this);},L.toArray=function(){return s.Kj(this);},L.Wb=function(t){return s.Lj(this,t);},Q.xdb=N(20),n.ek=function(t){var A,e,r;for(r=new h.Mkb(),e=new s.trb(t);e.a<e.c.a.length;)A=s.srb(e),h.Kkb(h.Kkb(r,iP+A.Lb()+'"'+A.Kb()+Ze),m1);return r.a;},n.fk=function(){s.Mj.call(this);},n.gk=function(t){var A,e;s.Mj.call(this);try{if(t)for(e=new s.trb(t);e.a<e.c.a.length;)A=s.srb(e),s.sj(this,A.Gb());}catch(r){if(r=M1(r),!Zt(r,19))throw Lt(r);}},n.hk=function(t){var A,e,r;if(s.Mj.call(this),!(t==null||w.UAb(t).length==0))for(A=0,e=h.jkb(t,X1(10));e!=-1;)r=n.Gi((w.OAb(A,e,w.UAb(t).length),w.UAb(t).substr(A,e-A))),r&&w.kAb(this.a,r),A=e+1,e=h.kkb(t,X1(10),A);},I(103,20,We,n.fk,n.gk,n.hk),L.pb=function(){return n.ek(this);},Q.l8=N(103),n.jk=function(t,A,e,r){this.b=t,this.a=A,this.c=e,this.d=r;},I(109,1,{109:1},n.jk),L.a=0,L.b=0,L.c=0,L.d=0,Q.m8=N(109),n.kk=function(t){var A,e,r,P,i,D;if(t.j!=0)return t.j;if(t.i&&n.Tl(t.e,t.a)!=15&&n.Tl(t.e,t.a)!=16){for(D=0;D<n.Lo(t.e,t.a);D++)if(i=n._o(t.e,t.a,D),n.Sm(t.e,i,t.a))return n.Zo(t.e,t.a,D)==t.b?t.j=n.im(t.e,i)==257?3:1:t.j=n.im(t.e,i)==257?1:3,t.j;}return A=n.Yl(t.e,t.a,t.g),r=n.Yl(t.e,t.a,t.b),r<A&&(r+=N1),n.Lo(t.e,t.a)==2?(e=r-A,e>3.0915926535897933&&e<3.191592653589793?(t.j=-1,t.j):(t.j=e<t1?4:2,t.j)):(P=n.Yl(t.e,t.a,t.d),P<A&&(P+=N1),t.j=P<r?2:4,t.j);},n.lk=function(t,A,e,r){var P,i,D,u;for(this.e=t,this.g=e,this.a=r,D=-1,u=0;u<n.Lo(this.e,this.a);u++){if(P=n.Zo(this.e,this.a,u),i=n._o(this.e,this.a,u),P==this.g){n.im(this.e,i)==386&&(this.j=-1);continue;}n.Sm(this.e,i,this.a)&&(this.i&&(t.u[r]|=H1),this.i=true),D==A[P]?(this.d=P,this.f=true,this.c=n.gq(this.e,i)):D<A[P]?(D=A[P],this.d=this.b,this.b=P):this.d=P;}},I(116,1,{},n.lk),L.a=0,L.b=0,L.c=false,L.d=0,L.f=false,L.g=0,L.i=false,L.j=0,Q.n8=N(116),n.mk=function(t,A,e,r){this.c=t,this.b=A,this.d=e,this.a=r;},I(18,1,{18:1},n.mk),L.pb=function(){return this.d;},L.a=0,L.c=0,Q.o8=N(18),n.ok=function(t){var A,e,r,P,i;if(P=new s.Wvb(new n.Hk()),i=new s.Tvb(),t.k!=null)for(r=0;r<t.k.length;r++)for(A=0;A<t.k[r].r;A++)s.Pvb(P,w.UAb(t.e[r].R.c*n.dm(t.k[r],A))),s.Pvb(i,s.Qvb(P)),P.b.a.length<i.b.a.length&&s.Pvb(P,s.Qvb(i));return e=P.b.a.length+i.b.a.length,e==0?n.pk(t):t.p.c*(e%2==0?(h.Tib(P.b.a.length==0?null:s.xj(P.b,0))+h.Tib(i.b.a.length==0?null:s.xj(i.b,0)))/2:h.Tib(P.b.a.length==0?null:s.xj(P.b,0)));},n.pk=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g;if(y=L0,T=oP,b=L0,g=oP,P=0,t.k!=null)for(u=0;u<t.k.length;u++)for(e=0;e<t.k[u].q;e++)U=n.Ai(t.e[u].R,n.Dl(t.k[u],e).a),c=n.Bi(t.e[u].R,n.Dl(t.k[u],e).b),y=d.Math.min(y,U),T=d.Math.max(T,U),b=d.Math.min(b,c),g=d.Math.max(g,c),++P;return P<=1?t.d:(i=T-y,D=g-b,o=(i+D)/2,l=3*P,A=i*D,r=o/(1-l),d.Math.sqrt(r*r-A/(1-l))-r);},n.qk=function(t){var A;if(t.p=new n.Ci(),t.k!=null)for(t.e=B(Q.Uab,C1,105,t.k.length,0,1),A=0;A<t.k.length;A++)t.e[A]=new ht.JN(t.k[A]);if(t.a!=null)for(t.b=B(Q.Uab,C1,105,t.a.length,0,1),A=0;A<t.a.length;A++)t.b[A]=new ht.JN(t.a[A]);t.d=24;},n.rk=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O;for(P=B(Q._ab,C1,35,t.k.length,0,1),O=0,E=0,c=0;c<t.k.length;c++)n.be(t.e[c],A,null,H1),P[c]=t.e[c].v,O+=P[c].b,E=d.Math.max(E,P[c].a);for(u=B(Q._ab,C1,35,t.a.length,0,1),R=0,S=0,b=0;b<t.a.length;b++)n.be(t.b[b],A,null,65552),u[b]=t.b[b].v,R=d.Math.max(R,u[b].b),S+=u[b].a+12;if(r=d.Math.max(48,R+24),E=d.Math.max(E,24+2*S),e=-1,!t.g)t.g=new n.fk(),s.sj(t.g,new tt.HH()),e=0;else {for(g=0;g<t.g.a.length;g++)if(Zt(s.xj(t.g,g),61)){e=g;break;}e==-1&&(e=t.g.a.length,s.sj(t.g,new tt.HH()));}for(F=18,T=0;T<t.k.length;T++){if(T==t.n){for(tt.GH(s.xj(t.g,e),F-18,E/2,F-18+r,E/2),i=F+-6,D=.5*(E-12)-S,k=0;k<t.a.length;k++)l=i+.5*(R-u[k].b)-u[k].c,y=D-u[k].d,n.dd(t.b[k],new n.Di(l,y)),D+=12+u[k].a;F+=r;}o=F-P[T].c,U=.5*(E-P[T].a)-P[T].d,n.dd(t.e[T],new n.Di(o,U)),F+=36+P[T].b;}t.o=false;},n.sk=function(t,A){var e,r;if(t.g)for(r=new s.trb(t.g);r.a<r.c.a.length;)e=s.srb(r),e.Ib(A,t.p);},n.tk=function(t,A){var e,r,P,i,D;if(t.i!=0&&t.k!=null){for(r=n.ok(t)/t.p.c,C.H3(A,t.i),C.F3(A,pt(1.6*r),true,false),i=0;i<t.k.length;i++)if(t.k[i].q!=0){for(P=new R1.Pgb(),e=0;e<t.k[i].q;e++)P.a=pt(P.a+n.Ql(t.k[i],e)),P.b=pt(P.b+n.Rl(t.k[i],e));P.a=P.a/t.k[i].q|0,P.b=P.b/t.k[i].q|0,P.a=pt(n.Ai(t.e[i].R,P.a)),P.b=pt(n.Bi(t.e[i].R,P.b)),D=t.c==0?""+(i+1):t.c==2?i<t.j?""+String.fromCharCode(65+i&n1):"R"+(i+1-t.j):t.c==1?i<t.n?""+String.fromCharCode(65+i&n1):"P"+(i+1-t.n):"?"+(i+1),C.o3(A,P.a,P.b,D);}}},n.uk=function(t,A){var e,r,P,i,D;if(t.e!=null)for(e=n.ok(t)/t.p.c,P=t.e,i=0,D=P.length;i<D;++i)r=P[i],n.Wd(r,t.f),r.r=e/r.R.c,n.Td(r,A);if(t.b!=null)for(P=t.b,i=0,D=P.length;i<D;++i)r=P[i],n.Td(r,A);},n.vk=function(t,A){t.d=A;},n.wk=function(t,A){t.f=A;},n.xk=function(t,A){var e,r,P,i;if(t.e!=null)for(r=t.e,P=0,i=r.length;P<i;++P)e=r[P],e.J=A;},n.yk=function(t,A,e){var r,P,i,D,u,o,l;if(t.e!=null)for(i=t.e,u=0,l=i.length;u<l;++u)r=i[u],n.Xd(r,A,e);if(t.b!=null)for(P=t.b,D=0,o=P.length;D<o;++D)r=P[D],n.Xd(r,A,e);},n.zk=function(t,A){t.i=A;},n.Ak=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k,F;if(n.Bk(t,A,e,r),n.zi(t.p))return null;if(t.k!=null)for(T=t.k,c=0,b=T.length;c<b;++c)y=T[c],n.vi(t.p,y);if(t.g)for(k=new s.trb(t.g);k.a<k.c.a.length;)g=s.srb(k),n.ti(t.p,g);if(t.e!=null)for(D=t.e,o=0,U=D.length;o<U;++o)P=D[o],n.yi(P.R);if(t.b!=null)for(i=t.b,u=0,l=i.length;u<l;++u)P=i[u],n.yi(P.R);return F=t.p,t.p=new n.Ci(),F;},n.Bk=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k;if(t.o&&n.rk(t,A),i=null,t.e!=null)for(u=t.e,l=0,y=u.length;l<y;++l)D=u[l],n.be(D,A,null,0),i=i?ht.oO(i,D.v):D.v;if(t.b!=null)for(u=t.b,l=0,y=u.length;l<y;++l)D=u[l],n.be(D,A,null,0),i=i?ht.oO(i,D.v):D.v;if(t.g)for(b=new s.trb(t.g);b.a<b.c.a.length;)c=s.srb(b),g=c.Jb(A),n.xi(t.p,g),i=i?ht.oO(i,g):g;if(!i)return null;if(P=n.ok(t),k=new n.Ei(i,e,P,r),!(k.c==1&&k.a==0&&k.b==0)){if(n.ui(k,t.p),t.e!=null)for(o=t.e,U=0,T=o.length;U<T;++U)D=o[U],n.dd(D,k);if(t.b!=null)for(u=t.b,l=0,y=u.length;l<y;++l)D=u[l],n.dd(D,k);return k;}return null;},n.Ck=function(t,A){t&&(this.k=B(Q.S8,GA,26,1,0,1),this.k[0]=t),this.c=0,this.g=A,n.qk(this);},n.Dk=function(t,A){var e,r;for(this.k=B(Q.S8,GA,26,t.g.a.length+t.f.a.length,0,1),r=0;r<t.g.a.length+t.f.a.length;r++)this.k[r]=r<t.g.a.length?s.xj(t.g,r):s.xj(t.f,r-t.g.a.length);for(this.n=t.g.a.length,this.a=B(Q.S8,GA,26,t.a.a.length,0,1),e=0;e<t.a.a.length;e++)this.a[e]=s.xj(t.a,e);this.o=false,this.c=1,this.g=A,n.qk(this);},n.Ek=function(t,A,e){this.k=t,this.c=2,this.g=e,this.j=A,n.qk(this);},n.Fk=function(t,A){this.k=t,this.c=0,this.g=A,n.qk(this);},n.Gk=function(t,A){return w.WAb((w.IAb(t),t))>w.WAb((w.IAb(A),A))?-1:w.WAb((w.IAb(t),t))<w.WAb((w.IAb(A),A))?1:0;},I(158,1,{},n.Ck,n.Dk,n.Ek,n.Fk),L.c=0,L.d=0,L.f=0,L.i=0,L.j=0,L.n=0,L.o=false,Q.q8=N(158),n.Hk=function(){},I(365,1,{},n.Hk),L.Rb=function(t,A){return n.Gk(t,A);},L.mb=function(t){return this===t;},Q.p8=N(365),n.Rk=function(){n.Rk=i1,n.Lk=j(v(Q.Qcb,1),yt,2,6,["?","H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr","Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og",p1,p1,p1,p1,p1,p1,p1,p1,p1,p1,"R4","R5","R6","R7","R8","R9","R10","R11","R12","R13","R14","R15","R16","R1","R2","R3","A","A1","A2","A3",p1,p1,"D","T","X","R","H2","H+","Nnn","HYD","Pol",p1,p1,p1,p1,p1,p1,p1,p1,p1,p1,p1,"Ala","Arg","Asn","Asp","Cys","Gln","Glu","Gly","His","Ile","Leu","Lys","Met","Phe","Pro","Ser","Thr","Trp","Tyr","Val"]),n.Pk=j(v(Q.wfb,1),Yt,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]),n.Ok=j(v(Q.O7,1),Y,6,15,[6]),n.Kk=j(v(Q.O7,1),Y,6,15,[2]),n.Mk=j(v(Q.O7,2),cA,10,0,[null,j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[0]),j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[4]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[0]),j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[4]),j(v(Q.O7,1),Y,6,15,[3,5]),j(v(Q.O7,1),Y,6,15,[2,4,6]),j(v(Q.O7,1),Y,6,15,[1,3,5,7]),j(v(Q.O7,1),Y,6,15,[0]),j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[2,4]),j(v(Q.O7,1),Y,6,15,[3,5]),j(v(Q.O7,1),Y,6,15,[2,4,6]),j(v(Q.O7,1),Y,6,15,[1,3,5,7]),j(v(Q.O7,1),Y,6,15,[0,2]),j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,j(v(Q.O7,1),Y,6,15,[1,2,3]),j(v(Q.O7,1),Y,6,15,[2,4]),j(v(Q.O7,1),Y,6,15,[3,5]),j(v(Q.O7,1),Y,6,15,[2,4,6]),j(v(Q.O7,1),Y,6,15,[1,3,5,7]),j(v(Q.O7,1),Y,6,15,[0,2,4,6]),j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[2])]),n.Nk=j(v(Q.O7,2),cA,10,0,[null,j(v(Q.O7,1),Y,6,15,[1]),null,j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),null,null,j(v(Q.O7,1),Y,6,15,[-3]),j(v(Q.O7,1),Y,6,15,[-2]),j(v(Q.O7,1),Y,6,15,[-1]),null,j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),null,j(v(Q.O7,1),Y,6,15,[-3]),j(v(Q.O7,1),Y,6,15,[-2]),j(v(Q.O7,1),Y,6,15,[-1]),null,j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2,3,4]),j(v(Q.O7,1),Y,6,15,[2,3,4,5]),j(v(Q.O7,1),Y,6,15,[2,3,6]),j(v(Q.O7,1),Y,6,15,[2,3,4,7]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[1,2]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2,4]),j(v(Q.O7,1),Y,6,15,[-3,3,5]),j(v(Q.O7,1),Y,6,15,[-2]),j(v(Q.O7,1),Y,6,15,[-1]),null,j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[4]),j(v(Q.O7,1),Y,6,15,[3,5]),j(v(Q.O7,1),Y,6,15,[6]),j(v(Q.O7,1),Y,6,15,[4,6,7]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2,4]),j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2,4]),j(v(Q.O7,1),Y,6,15,[-3,3,5]),j(v(Q.O7,1),Y,6,15,[-2,4,6]),j(v(Q.O7,1),Y,6,15,[-1]),null,j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3,4]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[4]),j(v(Q.O7,1),Y,6,15,[5]),j(v(Q.O7,1),Y,6,15,[6]),j(v(Q.O7,1),Y,6,15,[4,6,7]),j(v(Q.O7,1),Y,6,15,[3,4]),j(v(Q.O7,1),Y,6,15,[3,4]),j(v(Q.O7,1),Y,6,15,[2,4]),j(v(Q.O7,1),Y,6,15,[1,3]),j(v(Q.O7,1),Y,6,15,[1,2]),j(v(Q.O7,1),Y,6,15,[1,3]),j(v(Q.O7,1),Y,6,15,[2,4]),j(v(Q.O7,1),Y,6,15,[3,5]),j(v(Q.O7,1),Y,6,15,[-2,2,4]),j(v(Q.O7,1),Y,6,15,[-1,1]),null,j(v(Q.O7,1),Y,6,15,[1]),j(v(Q.O7,1),Y,6,15,[2]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[4]),j(v(Q.O7,1),Y,6,15,[4,5]),j(v(Q.O7,1),Y,6,15,[3,4,5,6]),j(v(Q.O7,1),Y,6,15,[3,4,5,6]),j(v(Q.O7,1),Y,6,15,[3,4,5,6]),j(v(Q.O7,1),Y,6,15,[3,4,5,6]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3,4]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[3]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[2,3]),j(v(Q.O7,1),Y,6,15,[3])]);},n.Sk=function(t,A){var e,r;e=-1,r=0,(t.G[A]&16)!=0&&(e=16,++r),(t.G[A]&64)!=0&&(e=64,++r),(t.G[A]&32)!=0&&(e=32,++r),(t.G[A]&4)!=0&&(e=4,++r),(t.G[A]&2)!=0&&(e=2,++r),(t.G[A]&8)!=0&&(e=8,++r),(t.G[A]&1)!=0&&(e=1,++r),e!=-1&&e!=(t.H[A]&127)&&(t.H[A]=e),r<2&&(t.G[A]&=-128);},n.Tk=function(t,A,e,r){var P;return P=n.Uk(t,6),n.ii(t.J[P],A,e,r),P;},n.Uk=function(t,A){return t.q>=t.M&&n.Rn(t,t.M*2),t.C[t.q]=0,n.Cn(t,t.q,A),t.s[t.q]=0,t.u[t.q]=0,t.B[t.q]=0,t.w[t.q]=0,n.ii(t.J[t.q],0,0,0),t.v!=null&&(t.v[t.q]=null),t.t!=null&&(t.t[t.q]=null),t.T=0,t.q++;},n.Vk=function(t,A,e){return n.Wk(t,A,e,n.Lm(t,A)||n.Lm(t,e)?16:1);},n.Wk=function(t,A,e,r){var P;if(A==e)return  -1;for(P=0;P<t.r;P++)if(t.D[0][P]==A&&t.D[1][P]==e||t.D[0][P]==e&&t.D[1][P]==A)return t.H[P]<r&&(t.H[P]=r),P;return n.Xk(t,A,e,r);},n.Xk=function(t,A,e,r){return t.r>=t.N&&n.Sn(t,t.N*2),t.D[0][t.r]=A,t.D[1][t.r]=e,t.H[t.r]=r,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},n.Yk=function(t,A){return n.Zk(t,A,A.q,A.r,true);},n.Zk=function(t,A,e,r,P){var i,D,u,o,l;for(t.K=t.K|A.K,D=B(Q.S7,H,6,A.q,15,1),o=n.cn(t,1),l=n.cn(t,2),i=0;i<e;i++)(P||S1(z(A.B[i],i0),0))&&(D[i]=n.ll(A,t,i,o,l));for(u=0;u<r;u++)(P||!n.Jm(A,A.D[0][u])&&!n.Jm(A,A.D[1][u]))&&n.ml(A,t,u,o,l,D[A.D[0][u]],D[A.D[1][u]],false);return t.L=t.L&&A.L,t.I=0,t.T=0,D;},n.$k=function(t,A,e,r,P,i,D,u){var o,l;return o=n.xl(t,A,e),o==-1?(t.q>=t.M&&n.Rn(t,t.M*2),o=n.Uk(t,r),n.ii(t.J[o],A,e,0),t.A[o]=P,n.gn(t,o,i),n.wn(t,o,D),n.mn(t,o,u),true):(l=n.fl(t,o,r,P,i,D),l=l|n.mn(t,o,u),l);},n._k=function(t,A,e,r){var P;for(P=0;P<t.r;P++)if(t.D[0][P]==A&&t.D[1][P]==e||t.D[0][P]==e&&t.D[1][P]==A)return n.il(t,P,r),t.T=0,P;return t.r>=t.N&&n.Sn(t,t.N*2),t.D[0][t.r]=A,t.D[1][t.r]=e,t.H[t.r]=r,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},n.al=function(t,A,e,r,P,i){for(var D,u,o;t.q+r>t.M;)n.Rn(t,t.M*2);for(;t.r+r>t.N;)n.Sn(t,t.N*2);return D=n.xl(t,A,e),D!=-1?n.bl(t,D,r,P,i):(u=n.yl(t,A,e),u!=-1?n.cl(t,u,r,P,i):(D=n.Tk(t,A,e,0),o=t1*(r-2)/r,n.Wm(t,D,r,D,P,0,t1-o,i),t.T=0,true));},n.bl=function(t,A,e,r,P){var i,D,u,o,l,U;if(r&&n.vp(t,A)>1||!r&&n.vp(t,A)>2)return  false;for(D=0,i=B(Q.Q7,jt,6,4,15,1),o=0;o<t.r;o++){for(l=0;l<2;l++)if(t.D[l][o]==A){if(D==2){D=3;break;}i[D++]=n.Yl(t,A,t.D[1-l][o]);}if(D==3)break;}return D==3?false:(U=D==1?i[0]+t1:d.Math.abs(i[0]-i[1])>t1?(i[0]+i[1])/2:(i[0]+i[1])/2+t1,u=t1*(e-2)/e,n.Wm(t,A,e,A,r,U-u/2,t1-u,P),t.T=0,true);},n.cl=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b,g;if(l=B(Q.S7,H,6,2,15,1),o=B(Q.Q7,jt,6,2,15,1),l[0]=t.D[0][A],l[1]=t.D[1][A],n.vp(t,l[0])>3||n.vp(t,l[1])>3)return  false;for(D=0,i=B(Q.Q7,jt,6,4,15,1),T=0;T<t.r;T++)if(T!=A){for(c=0;c<2;c++){for(b=0;b<2;b++)if(t.D[c][T]==l[b]){if(D==4){D=5;break;}i[D++]=n.Yl(t,l[b],t.D[1-c][T]);}if(D==5)break;}if(D==5)break;}if(D==5)return  false;for(o[0]=n.Yl(t,l[0],l[1]),o[0]<0?(o[1]=o[0]+t1,u=0):(o[1]=o[0],o[0]=o[1]-t1,u=1),g=0,y=0;y<D;y++)i[y]>o[0]&&i[y]<o[1]?--g:++g;return u=g>0?1-u:u,U=t1*(e-2)/e,n.Wm(t,l[u],e-1,l[1-u],r,o[g>0?0:1]+t1-U,t1-U,P),t.T=0,true;},n.dl=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g;for(P=B(Q.S7,H,6,A.q,15,1),o=n.cn(t,1),l=n.cn(t,2),r=0;r<A.q;r++)if(A.C[r]!=0||(A.t==null||A.t[r]==null?null:h.akb((c=A.t[r],O1.Enb(),c)))!=null)P[r]=n.ll(A,t,r,o,l);else if(P[r]=e,(A.T&8)!=0){for(u=0;u<A.r;u++)for(U=0;U<2;U++)if(A.D[U][u]==r){if(b=A.D[1-U][u],g=A.u[b]&3,y=false,g==1||g==2){for(i=0;i<A.r;i++)if(i!=u)for(T=0;T<2;T++)A.D[T][i]==b&&A.D[1-T][i]<r&&(y=!y);}y&&n.un(A,b,g==1?2:1,(A.u[b]&4)!=0);}}for(D=0;D<A.r;D++)n.ml(A,t,D,o,l,P[A.D[0][D]],P[A.D[1][D]],false);return t.L=t.L&&A.L,t.I=0,t.T=0,P;},n.el=function(t,A){var e,r,P,i,D,u,o,l,U;return e=t.J[A[0]],r=t.J[A[1]],P=t.J[A[2]],i=t.J[A[3]],o=new n.oi(r.a-e.a,r.b-e.b,r.c-e.c),l=new n.oi(P.a-r.a,P.b-r.b,P.c-r.c),U=new n.oi(i.a-P.a,i.b-P.b,i.c-P.c),D=new n.oi(o.b*l.c-o.c*l.b,-(o.a*l.c-o.c*l.a),o.a*l.b-o.b*l.a),u=new n.oi(l.b*U.c-l.c*U.b,-(l.a*U.c-l.c*U.a),l.a*U.b-l.b*U.a),-d.Math.atan2(d.Math.sqrt(l.a*l.a+l.b*l.b+l.c*l.c)*(o.a*u.a+o.b*u.b+o.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},n.fl=function(t,A,e,r,P,i){var D;return (e==1||e==151||e==152)&&n.vp(t,A)>1?false:(D=t.K&&Rt(z(t.B[A],-2),0),t.B[A]=z(t.B[A],-2),t.v!=null&&t.v[A]!=null&&(t.v[A]=null,D=true),t.t!=null&&t.t[A]!=null&&(t.t[A]=null,D=true),e==t.C[A]&&r==t.A[A]&&P==((t.u[A]&bA)>>>27)-1&&i==(t.u[A]&48)?D:((e==151||e==152)&&(r=e-149,e=1),t.u[A]&=960,t.C[A]=e,t.A[A]=r,t.s[A]=0,t.B[A]=0,n.gn(t,A,P),n.wn(t,A,i),n.an(t,t.w[A]),t.T=0,true));},n.gl=function(t,A,e,r){var P;return P=n.xl(t,A,e),P!=-1&&n.hl(t,P,r);},n.hl=function(t,A,e){if(e){if(t.s[A]>8)return  false;++t.s[A];}else {if(t.s[A]<-8)return  false;--t.s[A];}return t.T=0,true;},n.il=function(t,A,e){var r,P,i,D,u;return i=false,D=t.H[A],e==511?i=n.sm(t,A):n.Eq(t,A,e)&&(e==257||e==129?(r=n.Xm(t,A,t.D[0][A]),P=n.Xm(t,A,t.D[1][A]),e==D?(r==P||P)&&(u=t.D[0][A],t.D[0][A]=t.D[1][A],t.D[1][A]=u,i=true):(!r&&P&&(u=t.D[0][A],t.D[0][A]=t.D[1][A],t.D[1][A]=u),t.H[A]=e,i=true)):(t.H[A]=e,i=true)),i&&(t.T=(D&127)==(e&127)?t.T&7:0,t.G[A]=0),i;},n.jl=function(t){t.q=0,t.r=0,t.K=false,t.L=false,t.I=0,t.v=null,t.t=null,t.P=null,t.T=0;},n.kl=function(t){var A,e,r,P,i,D,u,o;for(D=0;D<t.r;D++)t.H[D]==512&&(e=t.D[0][D],r=t.D[1][D],t.C[e]==-1^t.C[r]==-1&&t.s[e]!=0&&t.s[r]!=0&&t.s[e]<0^t.s[r]<0&&(t.s[e]<0?(++t.s[e],--t.s[r]):(--t.s[e],++t.s[r])));for(o=B(Q.S7,H,6,t.q,15,1),P=0,A=0;A<t.q;A++){if(t.C[A]==-1){o[A]=-1;continue;}P<A&&(t.C[P]=t.C[A],t.s[P]=t.s[A],t.A[P]=t.A[A],t.u[P]=t.u[A],t.B[P]=t.B[A],t.w[P]=t.w[A],n.ji(t.J[P],t.J[A]),t.v!=null&&(t.v[P]=t.v[A]),t.t!=null&&(t.t[P]=t.t[A])),o[A]=P,++P;}for(t.q=P,u=0,i=0;i<t.r;i++)t.H[i]!=512&&(t.H[u]=t.H[i],t.F[u]=t.F[i],t.G[u]=t.G[i],t.D[0][u]=o[t.D[0][i]],t.D[1][u]=o[t.D[1][i]],++u);return t.r=u,o;},n.ll=function(t,A,e,r,P){var i,D,u;return i=A.q,i>=A.M&&n.Rn(A,A.M*2),u=(t.u[e]&l1)>>18,D=-1,u==1?r==-1?D=n.cn(A,u):D=d.Math.min(31,r+((t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&v0)>>20)):u==2&&(P==-1?D=n.cn(A,u):D=d.Math.min(31,P+((t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&v0)>>20))),A.C[i]=t.C[e],A.s[i]=t.s[e],A.A[i]=t.A[e],A.u[i]=t.u[e],A.B[i]=A.K?t.B[e]:0,n.ji(A.J[i],t.J[e]),A.w[i]=t.w[e],A.v!=null&&(A.v[i]=null),t.v!=null&&t.v[e]!=null&&A.K&&(A.v==null&&(A.v=B(Q.S7,E1,7,A.C.length,0,2)),A.v[i]=s.Brb(t.v[e],t.v[e].length)),A.t!=null&&(A.t[i]=null),t.t!=null&&t.t[e]!=null&&(A.t==null&&(A.t=B(Q.O7,cA,10,A.C.length,0,2)),A.t[i]=s.zrb(t.t[e],t.t[e].length)),D!=-1&&(A.u[i]&=-32505857,A.u[i]|=D<<20),++A.q,A.T=0,i;},n.ml=function(t,A,e,r,P,i,D,u){var o,l,U,y;return l=A.r,l>=A.N&&n.Sn(A,A.N*2),y=(t.F[e]&768)>>8,U=-1,y==1&&(r==-1?U=n.cn(A,y):U=d.Math.min(32,r+((t.F[e]&768)>>8!=1&&(t.F[e]&768)>>8!=2?-1:(t.F[e]&E0)>>10))),y==2&&(P==-1?U=n.cn(A,y):U=d.Math.min(32,P+((t.F[e]&768)>>8!=1&&(t.F[e]&768)>>8!=2?-1:(t.F[e]&E0)>>10))),A.D[0][l]=i,A.D[1][l]=D,o=u&&t.Zb(e)?8:t.H[e],A.H[l]=o,A.F[l]=t.F[e],A.G[l]=A.K?t.G[e]:0,U!=-1&&(A.F[l]&=-31745,A.F[l]|=U<<10),++A.r,A.T=0,l;},n.nl=function(t,A,e,r,P,i,D){return n.ml(t,A,e,r,P,i==null?t.D[0][e]:i[t.D[0][e]],i==null?t.D[1][e]:i[t.D[1][e]],D);},n.ol=function(t,A){var e,r;for(A.v=null,A.t=null,A.K=t.K,A.q=0,e=0;e<t.q;e++)n.ll(t,A,e,0,0);for(A.r=0,r=0;r<t.r;r++)n.ml(t,A,r,0,0,t.D[0][r],t.D[1][r],false);t.Yb(A);},n.pl=function(t,A){A.K=t.K,A.L=t.L,A.S=t.S,A.I=t.I,A.P=t.P,A.T=t.T&24;},n.ql=function(t,A){var e,r;for(e=0;e<t.r;e++)for(r=0;r<2;r++)t.D[r][e]==A&&(t.H[e]=512);n.an(t,t.w[A]),t.C[A]=-1,t.v!=null&&(t.v[A]=null),t.t!=null&&(t.t[A]=null),n.kl(t),t.T=0;},n.rl=function(t,A,e){var r,P;return r=n.xl(t,A,e),r!=-1?((t.u[r]&512)!=0?n.wl(t):n.ql(t,r),t.T=0,true):(P=n.yl(t,A,e),P!=-1?((t.u[t.D[0][P]]&t.u[t.D[1][P]]&512)!=0?n.wl(t):n.ul(t,P),t.T=0,true):false);},n.sl=function(t,A){var e,r,P,i;if(A.length==0)return null;for(r=A,P=0,i=r.length;P<i;++P)e=r[P],t.C[e]=-1;return n.vl(t);},n.tl=function(t,A){t.H[A]=512,n.kl(t),t.T=0;},n.ul=function(t,A){var e,r,P;for(r=0;r<2;r++){for(e=0,P=0;P<t.r;P++)P!=A&&(t.D[0][P]==t.D[r][A]||t.D[1][P]==t.D[r][A])&&++e;e==0&&(n.an(t,t.w[t.D[r][A]]),t.C[t.D[r][A]]=-1);}t.H[A]=512,n.kl(t),t.T=0;},n.vl=function(t){var A,e,r;for(r=false,A=0;A<t.q;A++)t.C[A]==-1&&(r=true,n.an(t,t.w[A]));for(e=0;e<t.r;e++)t.H[e]==512?r=true:(t.C[t.D[0][e]]==-1||t.C[t.D[1][e]]==-1)&&(t.H[e]=512,r=true);return r?(t.T=0,n.kl(t)):null;},n.wl=function(t){var A,e;for(e=false,A=0;A<t.q;A++)(t.u[A]&512)!=0&&(t.C[A]=-1,e=true);return e&&n.vl(t)!=null;},n.xl=function(t,A,e){var r,P,i,D,u,o,l,U;for(D=-1,P=n.Wl(t,t.q,t.r,n.Qk),u=L0,o=P*P/12,r=0;r<t.q;r++)l=t.J[r].a,U=t.J[r].b,i=(A-l)*(A-l)+(e-U)*(e-U),i<o&&i<u&&(u=i,D=r);return D;},n.yl=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S;for(T=-1,b=n.Wl(t,t.q,t.r,n.Qk),c=L0,r=0;r<t.r;r++)g=t.J[t.D[0][r]].a,F=t.J[t.D[0][r]].b,k=t.J[t.D[1][r]].a,S=t.J[t.D[1][r]].b,U=k-g,y=S-F,P=d.Math.sqrt(U*U+y*y),i=(g+k)/2,D=(F+S)/2,U=A-i,y=e-D,!(d.Math.sqrt(U*U+y*y)>P/2)&&(k==g?l=d.Math.abs(g-A):(u=(S-F)/(g-k),o=-u*g-F,l=d.Math.abs((u*A+e+o)/d.Math.sqrt(u*u+1))),l<b&&l<c&&(c=l,T=r));return T;},n.zl=function(t,A){return ((t.u[A]&bA)>>>27)-1;},n.Al=function(t,A){return (t.u[A]&49152)>>14;},n.Bl=function(t,A){return t.s[A];},n.Cl=function(t,A){return t.u[A]&448;},n.Dl=function(t,A){return t.J[A];},n.El=function(t,A){var e;return t.t==null||t.t[A]==null?null:h.akb((e=t.t[A],O1.Enb(),e));},n.Fl=function(t,A){return t.t==null?null:t.t[A];},n.Gl=function(t,A){return (t.u[A]&l1)>>18!=1&&(t.u[A]&l1)>>18!=2?-1:(t.u[A]&v0)>>20;},n.Hl=function(t,A){return (t.u[A]&l1)>>18;},n.Il=function(t,A){return n.Lk[t.C[A]];},n.Jl=function(t,A){return t.v==null?null:t.v[A];},n.Kl=function(t,A){var e,r,P;if(t.v==null||t.v[A]==null)return Rt(z(t.B[A],1),0)?"":n.Lk[t.C[A]];for(P="",r=0;r<t.v[A].length;r++)r>0&&(P=(w.IAb(P),P+(w.IAb(","),","))),e=t.v[A][r],P=h.$jb(P,n.Lk[e]);return P;},n.Ll=function(t,A){return d.Math.abs(t.w[A]);},n.Ml=function(t,A){return t.A[A];},n.Nl=function(t,A){return t.u[A]&3;},n.Ol=function(t,A){return t.B[A];},n.Pl=function(t,A){return t.u[A]&48;},n.Ql=function(t,A){return t.J[A].a;},n.Rl=function(t,A){return t.J[A].b;},n.Sl=function(t,A){return t.J[A].c;},n.Tl=function(t,A){return t.C[A];},n.Ul=function(t){return n.Wl(t,t.q,t.r,n.Qk);},n.Vl=function(t,A){return n.Wl(t,t.q,t.r,A);},n.Wl=function(t,A,e,r){return n.Xl(t,A,e,r,t.J);},n.Xl=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b;for(y=false,T=0,l=0;l<e;l++)t.H[l]!=16&&(t.G[l]&LA)==0&&++T;if(T==0){for(U=0;U<e;U++)(t.G[U]&LA)==0&&++T;y=true;}if(T==0){if(A<2)return r;for(b=L0,i=1;i<A;i++)for(D=0;D<i;D++)c=n.di(P[i],P[D]),c>0&&c<b&&(b=c);return b!=L0?.6*b:r;}for(u=0,o=0;o<e;o++)(y||t.H[o]!=16)&&(t.G[o]&LA)==0&&(u+=n.di(P[t.D[1][o]],P[t.D[0][o]]));return u/T;},n.Yl=function(t,A,e){return n.no(t.J[A].a,t.J[A].b,t.J[e].a,t.J[e].b);},n.Zl=function(t,A,e){return t.D[A][e];},n.$l=function(t,A){return ((t.G[A]&7680)>>9)+((t.G[A]&$e)>>13);},n._l=function(t,A){return (t.G[A]&7680)>>9;},n.am=function(t,A){return (t.F[A]&48)>>4;},n.bm=function(t,A){return (t.F[A]&768)>>8!=1&&(t.F[A]&768)>>8!=2?-1:(t.F[A]&E0)>>10;},n.cm=function(t,A){return (t.F[A]&768)>>8;},n.dm=function(t,A){var e,r,P,i;return e=t.D[0][A],r=t.D[1][A],P=t.J[r].a-t.J[e].a,i=t.J[r].b-t.J[e].b,d.Math.sqrt(P*P+i*i);},n.em=function(t,A,e){var r;for(r=0;r<t.r;r++)if((t.D[0][r]==A&&t.D[1][r]==e||t.D[0][r]==e&&t.D[1][r]==A)&&t.H[r]!=512)return r;return  -1;},n.fm=function(t,A){if(t.K&&(t.G[A]&127)!=0){if((t.G[A]&9)!=0)return 1;if((t.G[A]&2)!=0)return 2;if((t.G[A]&4)!=0)return 3;if((t.G[A]&32)!=0)return 4;if((t.G[A]&64)!=0)return 5;if((t.G[A]&16)!=0)return 0;}switch(t.H[A]&127){case 1:case 8:return 1;case 2:return 2;case 4:return 3;case 32:return 4;case 64:return 5;default:return 0;}},n.gm=function(t,A){return t.F[A]&3;},n.hm=function(t,A){return t.G[A];},n.im=function(t,A){return t.H[A];},n.jm=function(t,A){return t.H[A]&127;},n.km=function(t,A){var e,r,P,i,D;if(t.q==0)return null;for(r=t.J[0].a,i=t.J[0].b,P=t.J[0].a,D=t.J[0].b,e=1;e<t.q;e++)r>t.J[e].a?r=t.J[e].a:P<t.J[e].a&&(P=t.J[e].a),i>t.J[e].b?i=t.J[e].b:D<t.J[e].b&&(D=t.J[e].b);return A?(A.c=r,A.d=i,A.b=P-r,A.a=D-i):A=new ht.qO(r,i,P-r,D-i),A;},n.lm=function(t,A){var e;return e=t.C[A]<n.Mk.length?n.Mk[t.C[A]]:null,e==null?6:e[e.length-1];},n.mm=function(t,A,e){return n.nm(t,A,e,t.C[A]);},n.nm=function(t,A,e,r){var P,i;return r>=171&&r<=190?0:(i=0,(t.u[A]&48)==32&&(i-=1),((t.u[A]&48)==16||(t.u[A]&48)==48)&&(i-=2),P=t.s[A],P==0&&t.K&&(S1(z(t.B[A],mA),Qn)&&(P=-1),S1(z(t.B[A],mA),on)&&(P=1)),r==7||r==8||r==9?i+=P:r==6||r==14||r==32?i-=d.Math.abs(P):r==15||r==33?e-i-P<=3?i+=P:i-=P:r==16||r==34||r==52?e-i-P<=4?i+=P:i-=d.Math.abs(P):r==17||r==35||r==53?e-i-P<=5?i+=P:i-=d.Math.abs(P):i-=P,i);},n.om=function(t,A){var e;return e=n.pm(t,A),e+n.nm(t,A,e,t.C[A]);},n.pm=function(t,A){var e,r;return e=((t.u[A]&bA)>>>27)-1,e==-1&&(e=(r=t.C[A]<n.Mk.length?n.Mk[t.C[A]]:null,r==null?6:r[r.length-1])),e;},n.qm=function(t,A){var e,r,P,i,D;for(i=n.Tm(t,t.D[0][A])||n.Tm(t,t.D[1][A])?5:3,r=0;r<2;r++)e=t.D[r][A],P=n.fm(t,A)+(D=n.pm(t,e),D+n.nm(t,e,D,t.C[e]))-n.vp(t,e),i>P&&(i=P);return i;},n.rm=function(t,A){return (t.u[A]&H1)!=0;},n.sm=function(t,A){var e,r,P;return r=n.qm(t,A),e=n.Lm(t,t.D[0][A])||n.Lm(t,t.D[1][A]),P=e?16:1,t.H[A]==64?(t.H[A]=P,t.T=0,true):t.H[A]==32?(t.H[A]=r>4?64:P,t.T=0,true):t.H[A]==4?(t.H[A]=r>3?32:P,t.T=0,true):t.H[A]==2&&(t.H[A]=386,t.T&=7,(t.F[A]&128)==0)?true:t.H[A]==386?(r>2?t.H[A]=4:t.H[A]=P,t.T=0,true):(384&t.H[A])!=0?(t.H[A]=1,t.T&=7,true):!e&&r<2?false:t.H[A]==1?(t.H[A]=2,t.T=0,true):r<1?false:t.H[A]==16?(t.H[A]=1,t.T=0,true):false;},n.tm=function(t){var A;for(t.T=0,t.C=B(Q.S7,H,6,t.M,15,1),t.s=B(Q.S7,H,6,t.M,15,1),t.w=B(Q.S7,H,6,t.M,15,1),t.J=B(Q.j8,En,29,t.M,0,1),A=0;A<t.M;A++)t.J[A]=new n.ni();t.A=B(Q.S7,H,6,t.M,15,1),t.u=B(Q.S7,H,6,t.M,15,1),t.B=B(Q.T7,e0,6,t.M,14,1),t.v=null,t.t=null,t.D=T1(Q.S7,[E1,H],[7,6],15,[2,t.N],2),t.H=B(Q.S7,H,6,t.N,15,1),t.F=B(Q.S7,H,6,t.N,15,1),t.G=B(Q.S7,H,6,t.N,15,1);},n.um=function(t,A){t.T&=~A;},n.vm=function(t){var A;for(A=0;A<t.q;A++)if(t.J[A].c!=0)return  true;return  false;},n.wm=function(t,A){return (t.u[A]&Qe)!=0;},n.xm=function(t,A){return t.C[A]==-1;},n.ym=function(t,A){return (t.u[A]&4)!=0;},n.zm=function(t,A){return (t.u[A]&O0)!=0;},n.Am=function(t,A){return t.w[A]<0;},n.Bm=function(t,A){return (t.F[A]&UA)!=0;},n.Cm=function(t,A){return (t.G[A]&LA)!=0;},n.Dm=function(t,A){return (t.F[A]&H1)!=0;},n.Em=function(t,A){return t.H[A]==512;},n.Fm=function(t,A){return (t.F[A]&4)!=0;},n.Gm=function(t,A){return (t.F[A]&r0)!=0;},n.Hm=function(t,A){var e,r,P,i;if(t.K){if(Rt(z(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!n.qo(e))return  false;}}return n.qo(t.C[A]);},n.Im=function(t,A){var e,r,P,i;if(t.K){if(Rt(z(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!n.ro(e))return  false;}}return n.ro(t.C[A]);},n.Jm=function(t,A){return Rt(z(t.B[A],i0),0);},n.Km=function(t,A){return (t.u[A]&r0)!=0;},n.Lm=function(t,A){var e,r,P,i;if(t.K){if(Rt(z(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!(e>=3&&e<=4||e>=11&&e<=13||e>=19&&e<=31||e>=37&&e<=51||e>=55&&e<=84||e>=87&&e<=112))return  false;}}return n.so(t.C[A]);},n.Mm=function(t,A){return t.A[A]==0;},n.Nm=function(t,A){var e,r,P,i;if(t.K){if(Rt(z(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!(e==1||e>=5&&e<=9||e>=14&&e<=17||e>=32&&e<=35||e>=52&&e<=53))return  false;}}return n.uo(t.C[A]);},n.Om=function(t){var A;for(A=0;A<t.q;A++)switch(t.C[A]){case 1:case 5:case 6:case 7:case 8:case 9:case 14:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:continue;default:return  false;}return  true;},n.Pm=function(t,A){return (t.u[A]&512)!=0;},n.Qm=function(t,A){return (t.u[t.D[0][A]]&t.u[t.D[1][A]]&512)!=0;},n.Rm=function(t,A){return t.H[A]==257||t.H[A]==129;},n.Sm=function(t,A,e){return (t.H[A]==257||t.H[A]==129)&&t.D[0][A]==e;},n.Tm=function(t,A){var e,r,P,i;if(t.K){if(Rt(z(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!(e>=21&&e<=30||e>=39&&e<=48||e==57||e>=72&&e<=80||e==89||e>=104&&e<=112))return  false;}}return n.vo(t.C[A]);},n.Um=function(t,A){t.C[A]=-1;},n.Vm=function(t,A){t.H[A]=512;},n.Wm=function(t,A,e,r,P,i,D,u){var o,l,U,y,T,c,b,g,k,F;for(A!=r&&(k=t.J[A].a-t.J[r].a,F=t.J[A].b-t.J[r].b,u=d.Math.sqrt(k*k+F*F)),o=A,U=n.Wn(t,A)!=3,g=1;g<e;g++){for(T=t.J[o].a+u*d.Math.sin(i),c=t.J[o].b+u*d.Math.cos(i),b=-1,y=0;y<t.q;y++)if(d.Math.abs(T-t.J[y].a)<4&&d.Math.abs(c-t.J[y].b)<4){b=y;break;}b==-1&&(b=n.Tk(t,T,c,0),t.J[b].a=T,t.J[b].b=c,t.J[b].c=0),l=n.em(t,o,b),l==-1&&(l=n.Wk(t,o,b,n.Lm(t,o)||n.Lm(t,b)?16:1),P&&(U&&n.Wn(t,t.D[0][l])<4&&n.Wn(t,t.D[1][l])<3&&(t.H[l]=2),U=!U)),o=b,i+=D;}l=n.em(t,o,r),l==-1&&(l=n.Wk(t,o,r,n.Lm(t,o)||n.Lm(t,r)?16:1)),P&&U&&n.Wn(t,t.D[0][l])<4&&n.Wn(t,t.D[1][l])<4&&(t.H[l]=2);},n.Xm=function(t,A,e){var r,P;if(n.fm(t,A)!=1)return  false;if((t.u[e]&3)!=0)return  true;for(P=0;P<t.r;P++)if(P!=A&&t.H[P]==2&&(t.D[0][P]==e&&(t.u[t.D[1][P]]&3)!=0||t.D[1][P]==e&&(t.u[t.D[0][P]]&3)!=0))return  true;for(r=0;r<t.r;r++)if(r!=A&&t.H[r]==1&&(t.D[0][r]==e||t.D[1][r]==e)&&(t.F[r]&3)!=0)return  true;return  false;},n.Ym=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=-449;},n.Zm=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=ae;},n.$m=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=-513;},n._m=function(t){var A;for(A=0;A<t.r;A++)t.F[A]&=-98305;},n.an=function(t,A){var e;for(e=0;e<t.q;e++)d.Math.abs(t.w[e])==d.Math.abs(A)&&(t.w[e]=0);},n.bn=function(t){var A,e,r,P;for(P=false,e=0;e<t.q;e++)Rt(z(t.B[e],i0),0)&&(t.C[e]=-1,P=true);for(P&&n.vl(t),t.v!=null&&(t.v=null,P=true),A=0;A<t.q;A++)Rt(t.B[A],0)&&(t.B[A]=0,P=true);for(r=0;r<t.r;r++)t.G[r]!=0&&(t.G[r]=0,P=true),t.H[r]==8&&(t.H[r]=1,P=true);return P&&(t.T=0),P;},n.cn=function(t,A){var e,r,P,i,D,u,o,l,U;if(A==0)return 0;for(u=null,r=0;r<t.q;r++)(t.u[r]&l1)>>18==A&&(u==null&&(u=B(Q.xfb,Tt,6,32,16,1)),u[(t.u[r]&l1)>>18!=1&&(t.u[r]&l1)>>18!=2?-1:(t.u[r]&v0)>>20]=true);for(i=0;i<t.r;i++)(t.F[i]&768)>>8==A&&(u==null&&(u=B(Q.xfb,Tt,6,32,16,1)),u[(t.F[i]&768)>>8!=1&&(t.F[i]&768)>>8!=2?-1:(t.F[i]&E0)>>10]=true);if(U=0,u!=null){for(l=B(Q.S7,H,6,32,15,1),o=0;o<32;o++)u[o]&&(l[o]=U++);for(e=0;e<t.q;e++)(t.u[e]&l1)>>18==A&&(D=l[(t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&v0)>>20],t.u[e]&=-32505857,t.u[e]|=D<<20);for(P=0;P<t.r;P++)(t.F[P]&768)>>8==A&&(D=l[(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10],t.F[P]&=-31745,t.F[P]|=D<<10);}return U;},n.dn=function(t,A){var e;for(e=0;e<t.q;e++)t.J[e].a*=A,t.J[e].b*=A;},n.en=function(t,A){t.q=A,t.T=0;},n.fn=function(t,A){t.r=A,t.T=0;},n.gn=function(t,A,e){e>=-1&&e<=14&&(t.u[A]&=-2013265921,t.u[A]|=1+e<<27,t.C[A]==6&&(e==-1||e==0||e==2||e==4)&&(t.u[A]&=-49,e==2&&(t.u[A]|=16)));},n.hn=function(t,A,e){t.u[A]&=-49153,t.u[A]|=e<<14;},n.jn=function(t,A,e){t.s[A]=e,t.T=0;},n.kn=function(t,A,e){t.u[A]&=-449,t.u[A]|=e;},n.ln=function(t,A,e){e?t.u[A]|=Qe:t.u[A]&=tr,t.T&=7;},n.mn=function(t,A,e){var r,P,i;if(e!=null&&(w.UAb(e).length==0?e=null:(r=n.po(e,321),(r!=0&&r==t.C[A]||h.ckb(e,"?"))&&(n.Cn(t,A,r),e=null))),e==null){if(t.t!=null&&t.t[A]!=null)return t.t[A]=null,true;}else if(t.t==null&&(t.t=B(Q.O7,cA,10,t.M,0,2)),t.t[A]==null||!h.ckb(h.akb((i=t.t[A],O1.Enb(),i)),e))return t.t[A]=w.uAb((P=e,O1.Enb(),P)),true;return  false;},n.nn=function(t,A,e){e!=null&&e.length==0&&(e=null),e==null?t.t!=null&&(t.t[A]=null):(t.t==null&&(t.t=B(Q.O7,cA,10,t.M,0,2)),t.t[A]=e);},n.on=function(t,A,e,r){var P,i,D;if(e==0)t.u[A]&=Ar,t.u[A]|=e<<18;else {if(r>=32)return;if(r==-1){for(D=-1,i=0;i<t.q;i++)i!=A&&e==(t.u[i]&l1)>>18&&D<((t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&v0)>>20)&&(D=(t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&v0)>>20);for(P=0;P<t.r;P++)e==(t.F[P]&768)>>8&&D<((t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10)&&(D=(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10);if(r=D+1,r>=32)return;}t.u[A]&=Ar,t.u[A]|=e<<18|r<<20;}t.T&=7;},n.pn=function(t,A,e){t.v==null&&(t.v=B(Q.S7,E1,7,t.M,0,2)),w.nAb(e,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),t.v[A]=e,t.T=0,t.K=true;},n.qn=function(t,A,e,r){var P;if(e==null){t.v!=null&&(t.v[A]=null);return;}if(e.length==1&&!r){P=e[0],t.C[A]!=P&&n.fl(t,A,P,0,-1,0),t.v!=null&&(t.v[A]=null);return;}t.v==null&&(t.v=B(Q.S7,E1,7,t.M,0,2)),t.v[A]=e,r&&(t.B[A]=Pt(t.B[A],1)),t.T=0,t.K=true;},n.rn=function(t,A,e,r){t.w[A]=r?-e:e;},n.sn=function(t,A,e){e?t.u[A]|=r0:t.u[A]&=ae;},n.tn=function(t,A,e){t.A[A]=e,t.T&=t.C[A]==1?0:7;},n.un=function(t,A,e,r){t.u[A]&=-8,e!=3&&(t.u[A]&=tr),t.u[A]|=e,r&&(t.u[A]|=4);},n.vn=function(t,A,e,r){r?t.B[A]=Pt(t.B[A],e):t.B[A]=z(t.B[A],tA(e)),t.T=0,t.K=true;},n.wn=function(t,A,e){t.u[A]&=-49,t.u[A]|=e,t.T&=7;},n.xn=function(t,A,e){e?t.u[A]|=512:t.u[A]&=-513;},n.yn=function(t,A,e){t.u[A]&=-67108865,e&&(t.u[A]|=O0);},n.zn=function(t,A,e){t.J[A].a=e,t.T&=7;},n.An=function(t,A,e){t.J[A].b=e,t.T&=7;},n.Bn=function(t,A,e){t.J[A].c=e,t.T&=7;},n.Cn=function(t,A,e){e>=0&&e<=190&&(e==151||e==152?(t.C[A]=1,t.A[A]=e-149):(t.C[A]=e,t.A[A]=0),t.u[A]&=-2013265921,t.T=0);},n.Dn=function(t,A,e,r){t.D[A][e]=r,t.T=0;},n.En=function(t,A,e){e?t.F[A]|=UA:t.F[A]&=-32769;},n.Fn=function(t,A,e){t.F[A]&=-49,t.F[A]|=e<<4;},n.Gn=function(t,A,e,r){var P,i,D;if(e==0)t.F[A]&=-32513,t.F[A]|=e<<8;else {if(r>=32)return;if(r==-1){for(D=-1,i=0;i<t.q;i++)e==(t.u[i]&l1)>>18&&D<((t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&v0)>>20)&&(D=(t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&v0)>>20);for(P=0;P<t.r;P++)P!=A&&e==(t.F[P]&768)>>8&&D<((t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10)&&(D=(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10);if(r=D+1,r>=32)return;}t.F[A]&=-32513,t.F[A]|=e<<8|r<<10;}t.T&=7;},n.Hn=function(t,A,e){e?t.F[A]|=H1:t.F[A]&=-65537;},n.In=function(t,A,e){t.H[A]=e==1?1:e==2?2:e==3?4:16,t.T=0;},n.Jn=function(t,A,e,r){t.F[A]&=-131080,t.F[A]|=e,r&&(t.F[A]|=4);},n.Kn=function(t,A){t.F[A]|=r0;},n.Ln=function(t,A,e,r){r?t.G[A]|=e:t.G[A]&=~e,t.T=0,r&&e!=0&&(t.K=true);},n.Mn=function(t,A,e){t.H[A]=e,t.T=0;},n.Nn=function(t,A){t.I=A;},n.On=function(t,A){t.K!=A&&(t.K=A,A||n.bn(t),t.T=0);},n.Pn=function(t,A){t.T=A;},n.Qn=function(t,A){t.S=A;},n.Rn=function(t,A){var e,r;for(t.C=n.ho(t.C,A),t.s=n.ho(t.s,A),t.w=n.ho(t.w,A),r=t.J.length,t.J=n.jo(t.J,A),e=r;e<A;e++)t.J[e]=new n.ni();t.A=n.ho(t.A,A),t.u=n.ho(t.u,A),t.B=n.io(t.B,A),t.v!=null&&(t.v=n.lo(t.v,A)),t.t!=null&&(t.t=n.ko(t.t,A)),t.M=A;},n.Sn=function(t,A){t.D[0]=n.ho(t.D[0],A),t.D[1]=n.ho(t.D[1],A),t.H=n.ho(t.H,A),t.F=n.ho(t.F,A),t.G=n.ho(t.G,A),t.N=A;},n.Tn=function(t,A){t.O=A;},n.Un=function(t,A){t.P=A;},n.Vn=function(t,A){t.u[A]|=H1;},n.Wn=function(t,A){var e,r;for(r=0,e=0;e<t.r;e++)(t.D[0][e]==A||t.D[1][e]==A)&&(r+=n.fm(t,e));return r;},n.Xn=function(t){var A,e,r;for(e=false,r=false,A=0;A<t.q;A++)t.A[A]!=0&&(t.A[A]=0,e=true,t.C[A]==1&&(r=true));return r&&(t.T=0),e;},n.Yn=function(t,A,e){return n.Lm(t,A)||n.Lm(t,e)?16:1;},n.Zn=function(t,A,e){var r,P,i,D,u,o;for(D=t.C[A],t.C[A]=t.C[e],t.C[e]=D,D=t.s[A],t.s[A]=t.s[e],t.s[e]=D,D=t.A[A],t.A[A]=t.A[e],t.A[e]=D,D=t.u[A],t.u[A]=t.u[e],t.u[e]=D,o=t.B[A],t.B[A]=t.B[e],t.B[e]=o,D=t.w[A],t.w[A]=t.w[e],t.w[e]=D,i=t.J[A],t.J[A]=t.J[e],t.J[e]=i,t.v!=null&&(u=t.v[A],t.v[A]=t.v[e],t.v[e]=u),t.t!=null&&(u=t.t[A],t.t[A]=t.t[e],t.t[e]=u),r=0;r<t.r;r++)for(P=0;P<2;P++)t.D[P][r]==A?t.D[P][r]=e:t.D[P][r]==e&&(t.D[P][r]=A);t.T=0;},n.$n=function(t,A,e){var r;r=t.D[0][A],t.D[0][A]=t.D[0][e],t.D[0][e]=r,r=t.D[1][A],t.D[1][A]=t.D[1][e],t.D[1][e]=r,r=t.H[A],t.H[A]=t.H[e],t.H[e]=r,r=t.F[A],t.F[A]=t.F[e],t.F[e]=r,r=t.G[A],t.G[A]=t.G[e],t.G[e]=r,t.T=0;},n._n=function(t,A,e){var r;for(r=0;r<t.q;r++)t.J[r].a+=A,t.J[r].b+=e;t.U+=A,t.V+=e;},n.ao=function(t,A,e){var r,P;switch(P=e&127,r=n.qm(t,A),P){case 1:case 8:return r>=1;case 2:return r>=2;case 4:return r>=3;case 32:return r>=4;case 64:return r>=5;case 16:return  true;default:return  false;}},n.bo=function(t,A,e,r){var P,i,D;for(P=0;P<t.q;P++)(!r||(t.u[P]&512)!=0)&&(D=t.R[P]*A,i=t.Q[P]-e,t.J[P].a=t.U+D*d.Math.sin(i),t.J[P].b=t.V+D*d.Math.cos(i));r&&(t.T&=7);},n.co=function(t,A,e){var r,P,i;for(t.U=A,t.V=e,t.Q=B(Q.Q7,jt,6,t.q,15,1),t.R=B(Q.Q7,jt,6,t.q,15,1),r=0;r<t.q;r++)P=A-t.J[r].a,i=e-t.J[r].b,t.R[r]=d.Math.sqrt(P*P+i*i),t.Q[r]=n.no(A,e,t.J[r].a,t.J[r].b);},n.eo=function(){n.Rk(),this.M=this.N=256,n.tm(this);},n.fo=function(t,A){this.M=d.Math.max(1,t),this.N=d.Math.max(1,A),n.tm(this);},n.ho=function(t,A){var e;return e=B(Q.S7,H,6,A,15,1),h.Skb(t,0,e,0,d.Math.min(t.length,A)),e;},n.io=function(t,A){var e;return e=B(Q.T7,e0,6,A,14,1),h.Skb(t,0,e,0,d.Math.min(t.length,A)),e;},n.jo=function(t,A){var e,r;for(e=B(Q.j8,En,29,A,0,1),r=0;r<t.length;r++)t[r]&&(e[r]=new n.pi(t[r]));return e;},n.ko=function(t,A){var e,r;for(e=B(Q.O7,cA,10,A,0,2),r=0;r<t.length;r++)t[r]!=null&&(e[r]=B(Q.O7,Y,6,t[r].length,15,1),h.Skb(t[r],0,e[r],0,t[r].length));return e;},n.lo=function(t,A){var e,r;for(e=B(Q.S7,E1,7,A,0,2),r=0;r<t.length;r++)t[r]!=null&&(e[r]=B(Q.S7,H,6,t[r].length,15,1),h.Skb(t[r],0,e[r],0,t[r].length));return e;},n.mo=function(t){return n.Rk(),t>=0&&t<n.Mk.length&&n.Mk[t]!=null?n.Mk[t]:t>=171&&t<=190?n.Kk:n.Ok;},n.no=function(t,A,e,r){n.Rk();var P,i,D;return i=e-t,D=r-A,D!=0?(P=d.Math.atan(i/D),D<0&&(i<0?P-=t1:P+=t1)):P=i>0?F0:ue,P;},n.oo=function(t,A){n.Rk();var e;for(e=t-A;e<QP;)e+=N1;for(;e>t1;)e-=N1;return e;},n.po=function(t,A){n.Rk();var e,r,P,i,D;if((A&256)!=0&&h.ckb(t,"?"))return 0;for(r=1;r<=128;r++)if(!h.ckb(t,p1)&&h.dkb(t,n.Lk[r]))return r;if((A&2)!=0){for(P=129;P<=144;P++)if(h.dkb(t,n.Lk[P]))return P;}if((A&4)!=0){for(i=146;i<=148;i++)if(h.dkb(t,n.Lk[i]))return i;}if((A&1)!=0){for(D=151;D<=152;D++)if(h.dkb(t,n.Lk[D]))return D;}if((A&32)!=0&&h.dkb(t,n.Lk[153]))return 153;if((A&8)!=0&&h.dkb(t,n.Lk[154]))return 154;if((A&16)!=0&&h.dkb(t,n.Lk[145]))return 145;if((A&128)!=0&&h.dkb(t,n.Lk[159]))return 159;if((A&64)!=0){for(e=171;e<=190;e++)if(h.dkb(t,n.Lk[e]))return e;}return 0;},n.qo=function(t){switch(n.Rk(),t){case 7:case 8:case 9:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:return  true;}return  false;},n.ro=function(t){return n.Rk(),!(t==1||t==6||n.qo(t)||t==2||t==10||t==18||t==36||t==54||t==86||t>103);},n.so=function(t){return t>=3&&t<=4||t>=11&&t<=13||t>=19&&t<=31||t>=37&&t<=51||t>=55&&t<=84||t>=87&&t<=112;},n.to=function(t){return t==2||t==10||t==18||t==36||t==54||t==86;},n.uo=function(t){return t==1||t>=5&&t<=9||t>=14&&t<=17||t>=32&&t<=35||t>=52&&t<=53;},n.vo=function(t){return t>=21&&t<=30||t>=39&&t<=48||t==57||t>=72&&t<=80||t==89||t>=104&&t<=112;},I(88,1,{88:1,4:1},n.eo),L.Yb=function(t){n.pl(this,t);},L.Zb=function(t){return this.H[t]==8;},L.q=0,L.r=0,L.I=0,L.K=false,L.L=false,L.M=0,L.N=0,L.O=0,L.S=false,L.T=0,L.U=0,L.V=0,n.Qk=24,Q.y8=N(88),n.xo=function(t,A,e,r){var P,i,D,u,o,l,U,y,T;for(n.bw(A,1),r==null&&(r=B(Q.S7,H,6,A.q,15,1)),u=n.cn(t,1),o=n.cn(t,2),T=B(Q.xfb,Tt,6,A.q,16,1),l=B(Q.S7,H,6,A.q,15,1),l[0]=e,T[e]=true,r[e]=n.ll(A,t,e,u,o),D=0,U=0;D<=U;){for(y=0;y<n.Lo(A,l[D]);y++)i=A.i[l[D]][y],T[i]||(l[++U]=i,T[i]=true,r[i]=n.ll(A,t,i,u,o));++D;}for(P=0;P<A.r;P++)T[A.D[0][P]]&&n.ml(A,t,P,u,o,r==null?A.D[0][P]:r[A.D[0][P]],r==null?A.D[1][P]:r[A.D[1][P]],false);n.cn(t,1),n.cn(t,2),t.T=0;},n.yo=function(t){var A,e,r,P,i,D,u,o,l,U,y,T;for(t.j=B(Q.S7,H,6,t.q,15,1),t.e=B(Q.S7,H,6,t.q,15,1),t.i=B(Q.S7,E1,7,t.q,0,2),t.k=B(Q.S7,E1,7,t.q,0,2),t.n=B(Q.S7,E1,7,t.q,0,2),t.o=B(Q.S7,H,6,t.f,15,1),l=B(Q.S7,H,6,t.q,15,1),D=0;D<t.r;D++)++l[t.D[0][D]],++l[t.D[1][D]];for(r=0;r<t.q;r++)t.i[r]=B(Q.S7,H,6,l[r],15,1),t.k[r]=B(Q.S7,H,6,l[r],15,1),t.n[r]=B(Q.S7,H,6,l[r],15,1);for(y=false,u=0;u<t.g;u++){if(T=n.fm(t,u),T==0){y=true;continue;}for(U=0;U<2;U++)e=t.D[U][u],A=t.e[e],t.n[e][A]=T,t.i[e][A]=t.D[1-U][u],t.k[e][A]=u,++t.e[e],++t.j[e],e<t.f&&(T>1?t.o[e]+=T-1:t.H[u]==8&&(t.o[e]=1));}for(o=t.g;o<t.r;o++){if(T=n.fm(t,o),T==0){y=true;continue;}for(U=0;U<2;U++)e=t.D[U][o],A=t.e[e],t.n[e][A]=T,t.i[e][A]=t.D[1-U][o],t.k[e][A]=o,++t.e[e],t.D[1-U][o]<t.f&&++t.j[e];}if(y){for(A=B(Q.S7,H,6,t.q,15,1),P=0;P<t.q;P++)A[P]=t.e[P];for(i=0;i<t.r;i++)if(T=n.fm(t,i),T==0)for(U=0;U<2;U++)e=t.D[U][i],t.n[e][A[e]]=T,t.i[e][A[e]]=t.D[1-U][i],t.k[e][A[e]]=i,++A[e];}},n.zo=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x;for(n.bw(t,1),e&&(A=true),o=0;o<t.r;o++)if(l=n.fm(t,o),l==1||l==2){if(t.s[t.D[0][o]]>0&&t.s[t.D[1][o]]<0)i=t.D[0][o],D=t.D[1][o];else if(t.s[t.D[0][o]]<0&&t.s[t.D[1][o]]>0)i=t.D[1][o],D=t.D[0][o];else continue;if(n.Lm(t,i)||n.Lm(t,D)||t.C[i]<9&&n.vp(t,i)>3||t.C[D]<9&&n.vp(t,D)>3)continue;y=n.op(t,i)!=0,t.s[i]-=1,t.s[D]+=1,y||(S=t.H[o],l==1?t.H[o]=2:t.H[o]=4,(S==129||S==257)&&(x=t.D[0][o],F=n.pq(t,x,false),t.D[0][F]!=x&&(t.D[1][F]=t.D[0][F],t.D[0][F]=x))),t.T=0;}for(R=0,g=0,c=0,P=0;P<t.q;P++)R+=t.s[P],t.s[P]<0&&!n.Pp(t,P)&&(++g,n.Hm(t,P)&&(c-=t.s[P]));if(!A&&R!=0)throw Lt(new h.cC("molecule's overall charges are not balanced"));for(n.bw(t,1),E=0,O=e?R+c:c,u=0;u<t.q;u++)t.s[u]>0&&!n.Op(t,u)&&n.Hm(t,u)&&(U=d.Math.min(n.op(t,u),t.s[u]),U!=0&&O>=U&&(R-=U,E-=U,O-=U,t.s[u]-=U,t.T&=1));if(k=e?R:E,k<0){for(b=B(Q.S7,H,6,g,15,1),g=0,i=0;i<t.q;i++)t.s[i]<0&&!n.Pp(t,i)&&(b[g++]=(t.C[i]<<22)+i);for(w.nAb(b,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),T=b.length-1;k<0&&T>=b.length-g;T--)r=b[T]&W1,n.Hm(t,r)&&(U=d.Math.min(-k,-t.s[r]),R+=U,k+=U,t.s[r]+=U,t.T&=1);}return R;},n.Ao=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c;for(e=0;e<t.q;e++)c=n.pm(t,e),c+n.nm(t,e,c,t.C[e])-n.vp(t,e)<=0&&!(t.s[e]==0&&(t.C[e]==5||(P=t.C[e],P==7||P==15||P==33)||(r=t.C[e],r==8||r==16||r==34||r==52||r==84)))&&(t.B[e]=z(t.B[e],-6145));if(t.S)return  false;for(D=false,A=0;A<t.f;A++)if(u=t.e[A]-t.j[A],u>0)for(S1(z(t.B[A],B1),0)&&(U=S1(z(t.B[A],X0),896)?3:S1(z(t.B[A],X0),384)?2:S1(z(t.B[A],128),128)?1:0,o=(c=n.pm(t,A),c+n.nm(t,A,c,t.C[A])-n.vp(t,A)),t.s[A]==0&&S1(z(t.B[A],mA),0)&&t.C[A]!=6&&++o,y=u,y>3-U&&(y=3-U),y>o+u-U&&(y=o+u-U),y>0&&(T=U==0?0:c1(z(t.B[A],X0),y),T=Pt(T,Q.Mfb((y==3?7:u==2?3:1)<<7)),t.B[A]=z(t.B[A],-1921),t.B[A]=Pt(t.B[A],z(X0,T)))),l=t.j[A];l<t.e[A];l++)i=t.k[A][l],t.H[i]==1&&(t.C[t.i[A][l]]=-1,t.H[i]=512,D=true);return D&&n.kl(t),D;},n.Bo=function(t,A){var e,r,P,i;if(t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2){for(P=0;P<2;P++)if(e=n.Go(t,A,t.i[A][P]),e!=-1)for(i=0;i<t.j[e];i++)r=t.k[e][i],(t.H[r]==257||t.H[r]==129)&&t.D[0][r]==e&&(t.H[t.k[e][i]]=1);return;}if(t.o[A]==0||t.C[A]>=15)for(P=0;P<t.e[A];P++)r=t.k[A][P],(t.H[r]==257||t.H[r]==129)&&t.D[0][r]==A&&t.D[0][r]==A&&(t.H[r]=1);},n.Co=function(t,A,e,r,P){var i,D,u,o,l,U,y,T;for(n.bw(t,r?7:1),A.v=null,t.K&&n.On(A,true),o=e.length,P==null&&(P=B(Q.S7,H,6,o,15,1)),A.q=0,i=0;i<o;i++)if(e[i]){if(P[i]=n.ll(t,A,i,0,0),t.C[i]==1&&(T=n.vp(t,i),T>1)){for(y=0;y<t.e[i];y++)if(!e[t.i[i][y]]){n.gn(A,P[i],T);break;}}}else P[i]=-1;for(U=B(Q.S7,H,6,t.r,15,1),s.Lrb(U,U.length,-1),A.r=0,l=0;l<t.r;l++)D=t.D[0][l],u=t.D[1][l],D<o&&u<o&&(e[D]&&e[u]?U[l]=n.ml(t,A,l,0,0,P==null?t.D[0][l]:P[t.D[0][l]],P==null?t.D[1][l]:P[t.D[1][l]],r):t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(e[D]&&(A.s[P[D]]+=t.s[D]<0?1:-1),e[u]&&(A.s[P[u]]+=t.s[u]<0?1:-1)));n.pl(t,A),t.d&&(A.T=0),A.T=0,n.cn(A,1),n.cn(A,2),A.q!=o&&n.On(A,true),n.sq(t,A,o,P,U),n.rq(t,A,o,P),r&&n.Ge(new n.Ve(A),null,false);},n.Do=function(t,A,e,r,P){var i,D,u,o,l,U,y,T;for(n.bw(t,r?7:1),A.v=null,t.K&&n.On(A,true),P==null&&(P=B(Q.S7,H,6,t.q,15,1)),A.q=0,i=0;i<t.q;i++)for(P[i]=-1,U=0;U<t.j[i];U++)if(e[t.k[i][U]]){if(P[i]=n.ll(t,A,i,0,0),t.C[i]==1&&(T=n.vp(t,i),T>1)){for(y=0;y<t.e[i];y++)if(!e[t.k[i][y]]){n.gn(A,P[i],T);break;}}break;}for(l=B(Q.S7,H,6,t.r,15,1),A.r=0,o=0;o<t.r;o++)e[o]?l[o]=n.ml(t,A,o,0,0,P==null?t.D[0][o]:P[t.D[0][o]],P==null?t.D[1][o]:P[t.D[1][o]],r):(l[o]=-1,D=t.D[0][o],u=t.D[1][o],P[D]==-1^P[u]==-1&&t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(P[D]!=-1&&(A.s[P[D]]+=t.s[D]<0?1:-1),P[u]!=-1&&(A.s[P[u]]+=t.s[u]<0?1:-1)));return n.pl(t,A),t.d&&(A.T=0),A.T=0,n.cn(A,1),n.cn(A,2),A.q!=t.q&&n.On(A,true),n.sq(t,A,t.q,P,l),n.rq(t,A,t.q,P),r&&n.Ge(new n.Ve(A),null,false),P;},n.Eo=function(t,A){var e,r,P,i,D,u,o,l,U,y;if((A&~t.T)!=0&&((t.T&1)==0&&(n.Np(t),n.yo(t),t.T|=1,t.K&&!n.vm(t)&&n.Ao(t)&&(n.Np(t),n.yo(t),t.T|=1)),(A&~t.T)!=0)){if((t.T&-7)!=0){for(r=0;r<t.f;r++)t.u[r]&=-15369;for(i=0;i<t.g;i++)t.F[i]&=-705;if((A&4)==0){n.Jo(t,1),t.T|=2;return;}for(n.Jo(t,7),P=0;P<t.f;P++)for(U=0;U<t.j[P];U++)if(o=t.k[P][U],!(o<t.g&&n.pt(t.p,o)))for(u=t.i[P][U],y=0;y<t.j[u];y++)t.k[u][y]!=o&&t.n[u][y]>1&&(t.C[t.i[u][y]]==6?t.u[P]|=Z1:!n.Up(t,t.k[u][y])&&n.Hm(t,t.i[u][y])&&(t.u[P]|=U0));for(;;){for(l=false,e=0;e<t.f;e++)if(t.o[e]>0&&(t.u[e]&U0)!=0&&!n.ot(t.p,e)){for(U=0;U<t.j[e];U++)if(t.n[e][U]>1)for(u=t.i[e][U],o=t.k[e][U],y=0;y<t.j[u];y++)t.k[u][y]!=o&&(D=t.i[u][y],(t.u[D]&U0)==0&&(t.u[D]|=U0,l=true));}if(!l)break;}}t.T|=6;}},n.Fo=function(t,A){var e,r,P,i,D;if(e=-1,t.o[A]==1){for(i=0;i<t.j[A];i++)if(t.n[A][i]==2){if(r=t.i[A][i],t.j[r]==2&&t.o[r]==2){for(D=0;D<2;D++)if(P=t.i[r][D],P!=A&&t.o[P]==1){e=r;break;}}break;}}return e;},n.Go=function(t,A,e){var r,P;for(r=A;t.j[e]==2&&t.o[e]==2&&e!=r;)P=e,e=t.i[e][0]==A?t.i[e][1]:t.i[e][0],A=P;return e==r?-1:e;},n.Ho=function(t,A){var e;if(t.j[A]==3&&A<t.f&&n.ot(t.p,A)&&(t.p&&A<t.f?n.dt(t.p,A):0)>=5){for(e=0;e<t.j[A];e++)if(n.Vp(t,t.k[A][e]))return t.k[A][e];}return  -1;},n.Io=function(t,A,e,r,P){var i,D,u,o,l,U;if(n.bw(t,7),!((t.u[A]&x0)==0||e&&!(A<t.f&&n.ot(t.p,A))))for(o=B(Q.S7,H,6,t.f,15,1),o[0]=A,r[A]=true,u=0,l=0;u<=l;){for(U=0;U<t.j[o[u]];U++)D=t.k[o[u]][U],!P[D]&&(t.F[D]&64)!=0&&(!e||D<t.g&&n.pt(t.p,D))&&(P[D]=true,i=t.i[o[u]][U],r[i]||(r[i]=true,o[++l]=i));++u;}},n.Jo=function(t,A){var e,r,P,i,D,u,o,l;for(t.p=new n.Dt(t,A),r=B(Q.S7,H,6,t.f,15,1),P=0;P<t.g;P++)n.ft(t.p,P)!=0&&(t.F[P]|=64,++r[t.D[0][P]],++r[t.D[1][P]]);for(e=0;e<t.f;e++)r[e]==2?t.u[e]|=f0:r[e]==3?t.u[e]|=B1:r[e]>3&&(t.u[e]|=x0);for(l=0;l<t.p.j.a.length;l++)for(D=n.ht(t.p,l),o=n.it(t.p,l),u=D.length,i=0;i<u;i++)t.u[D[i]]|=8,t.F[o[i]]|=128,t.H[o[i]]==386&&(t.H[o[i]]=2);},n.Ko=function(t){var A,e,r,P,i,D,u,o;for(u=B(Q.xfb,Tt,6,t.q,16,1),e=0;e<t.q;e++)u[e]=t.C[e]==1&&t.A[e]==0&&t.s[e]==0&&(t.u[e]&bA)==0&&(t.t==null||t.t[e]==null);for(o=B(Q.xfb,Tt,6,t.q,16,1),D=0;D<t.r;D++){if(r=t.D[0][D],P=t.D[1][D],n.fm(t,D)!=1){u[r]=false,u[P]=false;continue;}o[r]&&(u[r]=false),o[P]&&(u[P]=false),u[r]&&n.Lm(t,P)&&t.C[P]!=13&&(u[r]=false),u[P]&&n.Lm(t,r)&&t.C[r]!=13&&(u[P]=false),o[r]=true,o[P]=true;}for(i=0;i<t.r;i++)u[t.D[0][i]]&&u[t.D[1][i]]&&(u[t.D[0][i]]=false,u[t.D[1][i]]=false);for(A=0;A<t.q;A++)o[A]||(u[A]=false);return u;},n.Lo=function(t,A){return t.e[A];},n.Mo=function(t,A){return t.i[A].length;},n.No=function(t,A){return t.e[A]-n.tp(t,A)+n.op(t,A);},n.Oo=function(t){var A,e;for(n.bw(t,7),A=0,e=0;e<t.p.j.a.length;e++)n.nt(t.p,e)&&++A;return A;},n.Po=function(t,A){var e,r;for(e=0,r=0;r<t.j[A];r++)n.Hm(t,t.i[A][r])&&!n.Cm(t,t.k[A][r])&&++e;return e;},n.Qo=function(t,A){return t.o[A];},n.Ro=function(t,A){return n.bw(t,7),t.o[A]==2&&t.j[A]==2?n.nq(t,A,false):n.pq(t,A,false);},n.So=function(t,A){var e;return e=t.u[A]&x0,e==0?0:e==f0?2:e==B1?3:4;},n.To=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b;for(n.bw(t,7),i=B(Q.xfb,Tt,6,t.g,16,1),y=B(Q.xfb,Tt,6,t.g,16,1),b=B(Q.S7,H,6,t.f,15,1),D=0,u=1;u<t.j[A];u++)if(r=t.k[A][u],(t.F[r]&64)!=0){for(l=0;l<u;l++)if(P=t.k[A][l],(t.F[P]&64)!=0&&(y[r]=true,y[P]=true,c=n.yp(t,b,t.i[A][u],t.i[A][l],e-2,null,y),y[r]=false,y[P]=false,c!=-1)){for(o=false,T=B(Q.S7,H,6,c,15,1),n.zp(t,b,T,c),U=0;U<c;U++)i[T[U]]||(i[T[U]]=true,o=true);o&&++D;}}return D;},n.Uo=function(t,A){return t.p&&A<t.f?n.dt(t.p,A):0;},n.Vo=function(t,A){return A?(n.bw(t,1),n.Wl(t,t.f,t.g,n.Qk)):n.Wl(t,t.q,t.r,n.Qk);},n.Wo=function(t){var A,e,r,P,i,D,u,o;for(n.bw(t,1),u=B(Q.R7,w0,6,t.f,15,1),r=B(Q.S7,H,6,t.f,15,1),o=0;o<t.f;o++){for(r[0]=o,P=B(Q.S7,H,6,t.f,15,1),P[o]=1,e=0,i=0;e<=i;){for(D=0;D<t.j[r[e]];D++)A=t.i[r[e]][D],P[A]==0&&(P[A]=P[r[e]]+1,r[++i]=A,u[o]+=P[A]-1);++e;}u[o]/=i;}return u;},n.Xo=function(t,A,e){var r;for(r=0;r<t.i[A].length;r++)if(t.i[A][r]==e)return t.k[A][r];return  -1;},n.Yo=function(t,A){return t.p&&A<t.g?n.ft(t.p,A):0;},n.Zo=function(t,A,e){return t.i[A][e];},n.$o=function(t,A){return t.j[A];},n._o=function(t,A,e){return t.k[A][e];},n.ap=function(t,A,e){return t.n[A][e];},n.bp=function(t,A){var e,r;if(e=0,t.K)for(r=0;r<t.j[A];r++)n.Jm(t,t.i[A][r])&&++e;return e;},n.cp=function(t,A){return t.e[A]-t.j[A];},n.dp=function(t,A,e,r,P){var i,D,u,o;if(D=t.e[A],t.o[A]!=0||A<t.f&&n.ot(t.p,A)||t.j[A]<3||D>4)return  false;for(o=B(Q.xfb,Tt,6,4,16,1),u=0;u<D;u++){if(i=3.9269908169872414-r[u],d.Math.abs(lP-i%F0)>.0872664675116539||(P[u]=3&pt(i/F0),o[P[u]]))return  false;if(o[P[u]]=true,(P[u]&1)==0){if(t.H[t.k[A][e[u]]]!=1)return  false;}else if(!n.Sm(t,t.k[A][e[u]],A))return  false;}return o[0]&&o[2];},n.ep=function(t,A,e,r,P){var i,D,u,o,l,U,y,T;if((t.p&&A<t.f?n.dt(t.p,A):0)>24||(i=t.e[A],P==null&&(P=B(Q.S7,H,6,i,15,1)),!n.dp(t,A,e,r,P)))return 3;for(o=-1,l=0;l<i;l++)if((P[l]&1)==1){if(D=t.H[t.k[A][e[l]]],o!=-1&&o!=D)return 3;o=D;}return U=d.Math.abs(P[0]-P[1])==2?1:0,u=P[U]-P[U+1],T=d.Math.abs(u)==3^P[U]<P[U+1],y=i==3||(P[3]&1)==1,T^y^o==129?1:2;},n.fp=function(t,A,e){var r,P,i,D;for(D=D=B(Q.xfb,Tt,6,t.q,16,1),i=n.gp(t,A,e,D),P=B(Q.S7,H,6,i,15,1),i=0,r=0;r<t.q;r++)D[r]&&(P[i++]=r);return P;},n.gp=function(t,A,e,r){var P,i,D,u,o,l,U;for(n.bw(t,1),r==null&&(r=B(Q.xfb,Tt,6,t.q,16,1)),o=B(Q.S7,H,6,t.q,15,1),o[0]=A,r[A]=true,D=0,l=0,u=1;D<=l;){for(i=e?n.Mo(t,o[D]):t.e[o[D]],U=0;U<i;U++)P=t.i[o[D]][U],P<r.length&&!r[P]&&(o[++l]=P,r[P]=true,++u);++D;}return u;},n.hp=function(t,A,e,r){var P,i,D,u,o,l,U,y,T;for(n.bw(t,1),i=0;i<t.q;i++)A[i]=-1;for(l=0,P=0;P<t.q;P++)if(A[P]==-1&&(!e||(t.u[P]&r0)!=0)){for(A[P]=l,U=B(Q.S7,H,6,t.q,15,1),U[0]=P,o=0,y=0;o<=y;){for(u=r?n.Mo(t,U[o]):t.e[U[o]],T=0;T<u;T++)D=t.i[U[o]][T],A[D]==-1&&(!e||(t.u[D]&r0)!=0)&&(U[++y]=D,A[D]=l);++o;}++l;}return l;},n.ip=function(t,A,e,r){var P,i,D,u,o,l,U,y,T;for(n.bw(t,1),i=0;i<t.q;i++)A[i]=-1;for(l=0,P=0;P<t.q;P++)if(A[P]==-1){for(A[P]=l,U=B(Q.S7,H,6,t.q,15,1),U[0]=P,o=0,y=0;o<=y;){for(u=r?n.Mo(t,U[o]):t.e[U[o]],T=0;T<u;T++)D=t.i[U[o]][T],A[D]==-1&&!e[t.k[U[o]][T]]&&(U[++y]=D,A[D]=l);++o;}++l;}return l;},n.jp=function(t,A){var e;return e=n.pm(t,A),e+n.nm(t,A,e,t.C[A])-n.vp(t,A);},n.kp=function(t,A){var e,r,P,i,D,u;for(i=B(Q.S7,H,6,t.q,15,1),r=0;r<t.q;r++)i[r]=r;P=t.q;do--P;while(P>=0&&A[P]);for(e=0;e<P;e++)if(A[i[e]]){u=i[e],i[e]=i[P],i[P]=u,D=A[e],A[e]=A[P],A[P]=D;do--P;while(A[P]);}return i;},n.lp=function(t){return n.kp(t,n.Ko(t));},n.mp=function(t,A,e){var r,P,i;if(P=n.vp(t,A),P-=n.nm(t,A,P,t.C[A]),e&&(P-=t.e[A]-t.j[A]),i=n.mo(t.C[A]),P<=i[0])return  -1;for(r=1;r<i.length;r++)if(i[r]>=P)return i[r];return P;},n.np=function(t){var A,e;if(t.K)return 0;for(n.bw(t,1),e=0,A=0;A<t.f;A++)e+=n.op(t,A);return e;},n.op=function(t,A){var e,r,P,i,D,u,o;if(t.K&&S1(z(t.B[A],B1),0)||!n.Aq(t,A))return 0;if(t.C[A]==0||h.ckb("*",t.t==null||t.t[A]==null?null:h.akb((i=t.t[A],O1.Enb(),i))))return t.e[A]==0?1:0;for(n.bw(t,1),u=0,P=0;P<t.e[A];P++)u+=t.n[A][P];if(t.K){for(e=1,r=0;r<t.j[A];r++)t.H[t.k[A][r]]==8&&++e;u+=e>>1;}if(u-=n.nm(t,A,u,t.C[A]),D=((t.u[A]&bA)>>>27)-1,D==-1)for(o=n.mo(t.C[A]),D=o[0],r=1;D<u&&r<o.length;r++)D=o[r];return d.Math.max(0,D-u);},n.pp=function(t,A){var e,r,P,i,D;if(!t.K||t.v==null||t.v[A]==null)return n.qp(t,A,t.C[A]);for(D=0,r=t.v[A],P=0,i=r.length;P<i;++P)e=r[P],D=d.Math.max(D,n.qp(t,A,e));return D;},n.qp=function(t,A,e){var r,P,i,D,u;if(i=n.vp(t,A),r=n.nm(t,A,i,e),D=((t.u[A]&bA)>>>27)-1,D==-1){for(u=n.mo(t.C[A]),P=0;P<u.length-1&&i>u[P]+r;)++P;D=u[P];}return D+r-i;},n.rp=function(t,A){return t.i[A].length-t.e[A];},n.sp=function(t){var A,e,r,P;for(n.bw(t,1),P=0,A=0;A<t.q;A++)r=t.A[A]!=0?t.A[A]:n.Pk[t.C[A]],P+=r+n.op(t,A)*n.Pk[1],t.C[A]>=171&&t.C[A]<=190&&(e=t.e[A],e>2&&(P-=(e-2)*n.Pk[1]));return P;},n.tp=function(t,A){var e,r;for(e=t.j[A],r=0;r<t.j[A];r++)t.C[t.i[A][r]]==1&&--e;return e;},n.up=function(t,A){return t.j[A]-n.bp(t,A);},n.vp=function(t,A){var e,r,P,i,D,u;for(t.$b(1),D=false,r=false,u=0,P=0;P<t.e[A];P++)(!t.K||!n.Jm(t,t.i[A][P]))&&(i=t.n[A][P],u+=i,i>1&&(D=true),e=t.k[A][P],t.H[e]==8&&(r=true));return r&&!D&&++u,u;},n.wp=function(t,A,e){var r,P,i;for(P=0,i=0;i<t.j[A];i++)r=t.i[A][i],r!=e&&t.j[r]>2&&++P;return P;},n.xp=function(t,A){var e,r,P,i,D,u,o,l,U,y;if(n.to(t.C[A]))return 0;if(t.C[A]==1)return t.j[A]==0?0:n.ro(t.C[t.i[A][0]])?-1:1;if(y=t.s[A],o=t.e[A]-n.tp(t,A)+n.op(t,A),t.C[A]!=1&&t.C[A]!=15&&(n.ro(t.C[A])?y+=o:y-=o),t.j[A]==0)return y;for(U=0;U<t.j[A];U++)i=t.i[A][U],e=n.Yp(t,t.k[A][U])?1:t.n[A][U],n.ro(t.C[A])?n.ro(t.C[i])||(y+=e):n.ro(t.C[i])?y-=e:t.C[A]!=t.C[i]&&(n.Xs(t.C[A]).a<n.Xs(t.C[i]).a?y+=e:y-=e);if(A<t.f&&n.tt(t.p,A)){for(r=0,P=0,u=0,l=0;l<t.j[A];l++)n.Yp(t,t.k[A][l])&&(D=t.C[t.i[A][l]],D==6?++P:n.qo(D)?++u:++r);t.C[A]==6?r>1?--y:u>1&&++y:n.Hm(t,A)?r+P>1&&--y:u+P>1&&++y;}return y;},n.yp=function(t,A,e,r,P,i,D){var u,o,l,U,y,T,c,b,g;if(e==r)return A[0]=e,0;for(t.$b(1),U=B(Q.S7,H,6,t.q,15,1),l=B(Q.S7,H,6,t.q,15,1),g=B(Q.S7,H,6,t.q,15,1),l[0]=e,U[e]=1,o=0,y=0;o<=y&&U[l[o]]<=P;){for(b=l[o],T=0;T<t.e[b];T++)if(D==null||D.length<=t.k[b][T]||!D[t.k[b][T]]){if(u=t.i[b][T],u==r){for(c=U[b],A[c]=u,A[--c]=b;c>0;)A[c-1]=g[A[c]],--c;return U[b];}U[u]==0&&(i==null||i.length<=u||!i[u])&&(l[++y]=u,U[u]=U[b]+1,g[u]=b);}++o;}return  -1;},n.zp=function(t,A,e,r){var P,i;for(n.bw(t,1),P=0;P<r;P++)for(i=0;i<t.e[A[P]];i++)if(t.i[A[P]][i]==A[P+1]){e[P]=t.k[A[P]][i];break;}},n.Ap=function(t,A,e){var r,P,i,D,u,o;if(A==e)return 0;for(n.bw(t,1),D=B(Q.S7,H,6,t.q,15,1),i=B(Q.S7,H,6,t.q,15,1),i[0]=A,D[A]=1,P=0,u=0;P<=u;){for(o=0;o<t.e[i[P]];o++){if(r=t.i[i[P]][o],r==e)return D[i[P]];D[r]==0&&(i[++u]=r,D[r]=D[i[P]]+1);}++P;}return  -1;},n.Bp=function(t,A,e,r,P){var i,D,u,o,l,U;if(A==e)return 0;for(t.$b(1),o=B(Q.S7,H,6,t.q,15,1),u=B(Q.S7,H,6,t.q,15,1),u[0]=A,o[A]=1,D=0,l=0;D<=l&&o[u[D]]<=r;){for(U=0;U<t.e[u[D]];U++){if(i=t.i[u[D]][U],i==e)return o[u[D]];o[i]==0&&(P==null||P.length<=i||!P[i])&&(u[++l]=i,o[i]=o[u[D]]+1);}++D;}return  -1;},n.Cp=function(t,A){return t.e[A]-t.j[A]+n.op(t,A);},n.Dp=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b;for(b=B(Q.S7,H,6,16,15,1),e=B(Q.Q7,jt,6,16,15,1),i=B(Q.Q7,jt,6,2,15,1),r=0,U=0;U<2;U++)for(P=t.D[U][A],y=0;y<t.j[P];y++)if(o=t.k[P][y],o!=A){if(r==4)return 0;u=t.i[P][y],b[r]=16,t.o[u]!=0&&(b[r]+=(t.u[u]&x0)!=0?1:4),(t.F[A]&64)!=0&&(t.F[o]&64)!=0&&(T=n.lt(t.p,A,o),T!=-1&&(b[r]+=n.nt(t.p,T)?64:6)),e[r++]=n.no(t.J[P].a,t.J[P].b,t.J[u].a,t.J[u].b);}for(i[0]=n.Yl(t,t.D[0][A],t.D[1][A]),i[0]<0?(i[1]=i[0]+t1,D=false):(i[1]=i[0],i[0]=i[1]-t1,D=true),c=0,l=0;l<r;l++)e[l]>i[0]&&e[l]<i[1]?c-=b[l]:c+=b[l];return D?-c:c;},n.Ep=function(t){return n.bw(t,7),t.p;},n.Fp=function(t){var A,e,r,P,i,D,u,o,l;for(l=0,n.bw(t,7),r=0;r<t.g;r++)if(n.fm(t,r)==1&&(t.F[r]&64)==0){for(u=true,D=0;D<2;D++){if(A=t.D[D][r],t.j[A]==1){u=false;break;}if(t.C[A]==7&&!(A<t.f&&n.ot(t.p,A))){for(e=t.D[1-D][r],o=0;o<t.j[e];o++)if(P=t.i[e][o],i=t.k[e][o],i!=r&&n.fm(t,i)>1&&!(P<t.f&&n.ot(t.p,P))&&n.Hm(t,P)){u=false;break;}}}u&&!n.cq(t,r)&&++l;}return l;},n.Gp=function(t,A,e){var r,P,i,D,u,o;for(P=0,i=0;i<t.j[A];i++)if(r=t.i[A][i],r!=e){for(D=0,u=0;u<t.j[r];u++)o=t.i[r][u],o!=A&&n.Up(t,t.k[r][u])&&t.j[o]>2&&++D;D==2&&++P;}return P;},n.Hp=function(t,A){var e,r,P,i;for(e=t.e[A],i=B(Q.S7,H,6,e,15,1),P=0;P<e;P++)i[P]=(t.i[A][P]<<16)+P;for(w.nAb(i,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),r=0;r<e;r++)i[r]&=n1;return i;},n.Ip=function(t,A){var e,r;if(n.bw(t,1),t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2){for(e=0;e<2;e++)for(r=0;r<t.e[t.i[A][e]];r++)if(n.Sm(t,t.k[t.i[A][e]][r],t.i[A][e]))return t.k[t.i[A][e]][r];}else for(e=0;e<t.e[A];e++)if(n.Sm(t,t.k[A][e],A))return t.k[A][e];return  -1;},n.Jp=function(t,A,e){return n.fm(t,A)!=1?0:16-t.e[e]+16*d.Math.max(0,d.Math.min(9,t.p&&A<t.g?n.ft(t.p,A):0)-2)+((t.H[A]&384)==0||t.D[0][A]!=e?UA:0)+((t.u[e]&3)==0?Z1:0)+(t.C[e]==1?B1:0)+(t.e[e]==1?f0:0)+((t.F[A]&64)!=0?0:512)+(t.C[e]!=6?256:0);},n.Kp=function(t,A,e,r,P,i){var D,u,o,l,U;for(n.bw(t,1),P&&(n.jl(P),P.d=null,P.K=false),o=B(Q.S7,H,6,t.q,15,1),r==null?r=B(Q.xfb,Tt,6,t.q,16,1):s.Prb(r,r.length),o[0]=A,o[1]=e,r[A]=true,r[e]=true,u=1,l=1;u<=l;){for(U=0;U<t.e[o[u]];U++){if(D=t.i[o[u]][U],D==A&&u!=1)return  -1;r[D]||(r[D]=true,o[++l]=D);}++u;}return P&&(i==null&&(i=B(Q.S7,H,6,r.length,15,1)),n.Co(t,P,r,false,i),n.fl(P,i[A],0,0,-1,0)),r[A]=false,l;},n.Lp=function(t,A,e){var r,P,i,D,u,o;for(n.bw(t,1),i=B(Q.S7,H,6,t.f,15,1),o=B(Q.xfb,Tt,6,t.f,16,1),i[0]=A,i[1]=e,o[A]=true,o[e]=true,P=1,D=1;P<=D;){for(u=0;u<t.j[i[P]];u++){if(r=t.i[i[P]][u],r==A&&P!=1)return  -1;o[r]||(o[r]=true,i[++D]=r);}++P;}return D;},n.Mp=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c;if(n.fm(t,e)!=2&&!(e<t.g&&n.pt(t.p,e))||(c=t.F[e]&3,c!=1&&c!=2))return  -1;for(o=0;o<2;o++){for(r=t.D[o][e],P=t.D[1-o][e],T=-1,D=false,U=0;U<t.j[r];U++)i=t.i[r][U],i!=P&&(i==A?D=true:T=i);if(D){for(y=-1,u=-1,l=0;l<t.j[P];l++)i=t.i[P][l],i!=r&&(y==-1?y=i:i>y?u=i:(u=y,y=i));return t.j[r]==2?t.j[P]==2?c==2?y:-1:c==2?y:u:t.j[P]==2?c==2^A<T?-1:y:c==2^A<T?u:y;}}return  -1;},n.Np=function(t){var A,e,r,P,i,D,u,o,l,U;D=n.Ko(t),o=t.q;do--o;while(o>=0&&D[o]);for(A=0;A<o;A++)if(D[A]){n.Zn(t,A,o),U=D[A],D[A]=D[o],D[o]=U;do--o;while(D[o]);}if(t.f=o+1,t.q==t.f){t.g=t.r;return;}for(u=B(Q.xfb,Tt,6,t.r,16,1),i=0;i<t.r;i++)e=t.D[0][i],r=t.D[1][i],(D[e]||D[r])&&(u[i]=true);l=t.r;do--l;while(l>=0&&u[l]);for(P=0;P<l;P++)if(u[P]){n.$n(t,P,l),u[P]=false;do--l;while(u[l]);}t.g=l+1;},n.Op=function(t,A){var e;for(e=0;e<t.j[A];e++)if(t.s[t.i[A][e]]<0)return  true;return  false;},n.Pp=function(t,A){var e;for(e=0;e<t.j[A];e++)if(t.s[t.i[A][e]]>0)return  true;return  false;},n.Qp=function(t,A){var e;return e=t.C[A],e==3||e==11||e==19||e==37||e==55||e==87;},n.Rp=function(t,A){return (t.u[A]&Z1)!=0;},n.Sp=function(t,A){var e,r,P,i,D,u;for(n.bw(t,1),D=0;D<2;D++)if(e=t.D[D][A],t.C[e]==7){for(r=t.D[1-D][A],u=0;u<t.j[r];u++)if(P=t.i[r][u],i=t.k[r][u],(t.C[P]==7||t.C[P]==8||t.C[P]==16)&&n.fm(t,i)>=2)return  true;}return  false;},n.Tp=function(t,A){return A<t.f&&n.ot(t.p,A);},n.Up=function(t,A){return A<t.g&&n.pt(t.p,A);},n.Vp=function(t,A){var e,r,P,i,D,u;return t.H[A]!=1||A<t.g&&n.pt(t.p,A)||(t.F[A]&64)!=0&&(t.p&&A<t.g?n.ft(t.p,A):0)<7||(e=t.D[0][A],!(e<t.f&&n.ot(t.p,e))||(t.p&&e<t.f?n.dt(t.p,e):0)<5)||(r=t.D[1][A],!(r<t.f&&n.ot(t.p,r))||(t.p&&r<t.f?n.dt(t.p,r):0)<5)?false:(P=n.wp(t,e,r),i=n.wp(t,r,e),(t.p&&e<t.f?n.dt(t.p,e):0)>5&&(t.p&&r<t.f?n.dt(t.p,r):0)>5?P+i>2:(D=n.Gp(t,e,r),u=n.Gp(t,r,e),P==2&&u>=1||i==2&&D>=1||D==2&&(i>=1||u>=1)||u==2&&(P>=1||D>=1)));},n.Wp=function(t,A){return t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2&&t.C[A]<=7;},n.Xp=function(t,A){var e;return e=t.C[A],e==8||e==16||e==34||e==52||e==84;},n.Yp=function(t,A){return A<t.g&&(n.ut(t.p,A)||t.H[A]==8);},n.Zp=function(t,A){var e;return e=t.C[A],e==4||e==12||e==20||e==38||e==56||e==88;},n.$p=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g;if(t.C[A]!=7||t.j[A]==4)return  false;if(A<t.f&&n.ot(t.p,A)||t.o[A]!=0||Rt(z(t.B[A],NA),0))return  true;if(t.s[A]==1)return  false;for(o=0;o<t.j[A];o++)if(n.Yo(t,t.k[A][o])==3)return  false;for(D=0,l=0;l<t.j[A];l++)t.n[A][l]==1&&(r=t.C[t.i[A][l]],(r==8||r==9||r==17)&&++D);if(D==0){for(u=0;u<t.j[A];u++)if(P=t.i[A][u],t.o[P]!=0){if(P<t.f&&n.ot(t.p,P)){if((t.p&&P<t.f?n.dt(t.p,P):0)>=5){for(g=0,T=0;T<t.j[P];T++)b=t.i[P][T],b!=A&&n.tp(t,b)>=3&&++g;if(c=n.tp(t,A),g==2&&c>=2||g==1&&c==3)continue;}return !e||!n.eq(t,A);}for(y=0;y<t.j[P];y++)if(t.n[P][y]==2||n.Up(t,t.k[P][y]))return !e||!n.eq(t,A);}}if(D<2)for(u=0;u<t.j[A];u++){for(P=t.i[A][u],U=false,i=false,y=0;y<t.j[P];y++)t.i[P][y]!=A&&(t.n[P][y]!=1&&(t.C[t.i[P][y]]==7||t.C[t.i[P][y]]==8||t.C[t.i[P][y]]==16)&&(U=true),t.n[P][y]==1&&t.C[t.i[P][y]]==7&&(i=true));if(U&&(!i||D==0))return !e||!n.eq(t,A);}return  false;},n._p=function(t,A){var e;return e=t.C[A],e==9||e==17||e==35||e==53||e==85;},n.aq=function(t,A){return A<t.f&&n.vt(t.p,A);},n.bq=function(t,A){var e;return e=t.C[A],e==7||e==15||e==33;},n.cq=function(t,A){var e,r,P,i,D,u;if(n.fm(t,A)!=1)return  false;for(i=0;i<2;i++){for(e=t.D[i][A],u=t.D[1-i][A];t.o[e]==2&&t.j[e]==2&&t.C[e]<10;)for(D=0;D<2;D++)if(r=t.i[e][D],r!=u){if(t.j[r]==1||(P=t.k[e][D],n.fm(t,P)==1&&P<A))return  true;u=e,e=r;break;}if(t.j[e]==1)return  true;}return  false;},n.dq=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b,g,k,F,S;if(T=B(Q.xfb,Tt,6,t.g,16,1),T[P]=true,c=B(Q.S7,H,6,11,15,1),b=n.yp(t,c,r,A,10,null,T),b==-1)return  false;for(D=1;!n.qt(t.p,e,c[D]);)++D;if(i=b-D,u=c[D],S=n.kt(t.p,e),S==6&&i==2&&D==3&&n.So(t,c[1])>=3){for(F=n.ht(t.p,e),y=0;y<6;y++)if(A==F[y]){if(k=n.Ct(t.p,e,u==F[n.Ct(t.p,e,y+2)]?y-2:y+2),g=F[k],U=t.u[g]&x0,(U==0?0:U==f0?2:U==B1?3:4)>=3&&n.Bp(t,c[1],g,2,null)==2)return  true;break;}}if(o=t.o[u]==1||u<t.f&&n.ot(t.p,u)||n.$p(t,u,false),l=!o&&t.C[u]==7&&t.s[u]!=1,i==1)return !o&&!l&&S<=4&&D<=3;switch(S){case 4:if(!o&&!l&&D<=4)return  true;break;case 5:if(l){if(D<=3)return  true;}else if(!o&&D<=4)return  true;break;case 6:if(i==2){if(o){if(D<=4)return  true;}else if(!l&&D<=3)return  true;}else if(i==3){if(o){if(D<=6)return  true;}else if(D<=4)return  true;}break;case 7:if(i==3&&D<=3)return  true;}return  false;},n.eq=function(t,A){var e,r,P,i,D;if(A<t.f&&n.ot(t.p,A)||t.o[A]!=0||Rt(z(t.B[A],NA),0)||(r=t.u[A]&x0,(r==0?0:r==f0?2:r==B1?3:4)<3)||(D=t.p&&A<t.f?n.dt(t.p,A):0,D>7))return  false;for(i=0;i<t.p.j.a.length&&!(n.kt(t.p,i)==D&&n.qt(t.p,i,A));)++i;if(i>=f0&&i==t.p.j.a.length)return  false;for(P=0;P<t.j[A];P++)if(e=t.k[A][P],!n.rt(t.p,i,e)&&n.dq(t,A,i,t.i[A][P],e))return  true;return  false;},n.fq=function(t,A){return (t.u[A]&x0)!=0;},n.gq=function(t,A){return (t.F[A]&64)!=0;},n.hq=function(t,A){return t.C[A]==1&&t.A[A]==0&&t.s[A]==0&&(t.u[A]&bA)==0&&(t.t==null||t.t[A]==null);},n.iq=function(t,A){return (t.u[A]&8)!=0;},n.jq=function(t,A){return (t.F[A]&128)!=0;},n.kq=function(t,A){return (t.u[A]&U0)!=0;},n.lq=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(n.bw(t,1),n.mq(t),l=false,e=0;e<t.f;e++)if(t.C[e]==7&&t.s[e]==0){if(y=n.vp(t,e),y==4){for(U=0;U<t.j[e];U++)if(u=t.i[e][U],t.n[e][U]==1&&t.C[u]==8&&t.j[u]==1&&t.s[u]==0){l=true,++t.s[e],--t.s[u];break;}}else if(y==5)for(U=0;U<t.j[e];U++){if(u=t.i[e][U],o=t.k[e][U],t.n[e][U]==2&&t.C[u]==8){l=true,++t.s[e],--t.s[u],t.H[o]=1;break;}if(t.n[e][U]==3&&t.C[u]==7){l=true,++t.s[e],--t.s[u],t.H[o]=2;break;}}}for(D=false,i=0;i<t.g;i++)for(U=0;U<2;U++)if(n.Hm(t,t.D[U][i])){A=t.D[1-U][i],P=t.C[A],(P==3||P==11||P==19||P==37||P==55||P==87||(r=t.C[A],r==4||r==12||r==20||r==38||r==56||r==88))&&(n.fm(t,i)==1?(++t.s[A],--t.s[t.D[U][i]],t.H[i]=512,D=true):t.H[i]==16&&(t.H[i]=512,D=true));break;}return D&&(n.kl(t),l=true),l&&(t.T=0),l;},n.mq=function(t){var A;for(A=0;A<t.g;A++)if(t.H[A]==8)return n.Ge(new n.Ve(t),null,false);return  false;},n.nq=function(t,A,e){var r,P,i,D,u,o,l,U;for(l=-1,P=0,u=0;u<2;u++)for(r=t.i[A][u],o=0;o<t.e[r];o++)i=t.i[r][o],i!=A&&(D=t.k[r][o],U=n.Jp(t,D,i),P<U&&(!e||!(t.H[D]==257||t.H[D]==129))&&(P=U,l=D));return l;},n.oq=function(t,A){var e,r,P,i,D,u,o,l;for(o=-1,r=0,D=0;D<2;D++)for(e=t.D[D][A],u=0;u<t.e[e];u++)P=t.i[e][u],P!=t.D[1-D][A]&&(i=t.k[e][u],l=n.Jp(t,i,P),r<l&&(r=l,o=i));return o;},n.pq=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S;for(r=t.e[A],P=B(Q.Q7,jt,6,r,15,1),c=0;c<r;c++)P[c]=n.Yl(t,A,t.i[A][c]);for(b=1;b<r;b++)for(g=0;g<b;g++)if(i=d.Math.abs(n.oo(P[b],P[g])),i<.08||i>z8){for(D=0,u=0,k=0;k<r;k++)k!=b&&k!=g&&(D+=d.Math.abs(Ut.o4(P[b],P[k])),u+=d.Math.abs(Ut.o4(P[g],P[k])));if(l=D<u?t.k[A][b]:t.k[A][g],n.fm(t,l)==1&&(!e||!(t.H[l]==257||t.H[l]==129)))return l;}for(F=-1,o=0,T=0;T<r;T++)U=t.i[A][T],y=t.k[A][T],S=n.Jp(t,y,U),o<S&&(!e||!(t.H[y]==257||t.H[y]==129))&&(o=S,F=y);return F;},n.qq=function(t,A){var e,r,P,i,D,u;for(n.bw(t,A?15:1),t.q=t.f,t.r=t.g,u=B(Q.xfb,Tt,6,t.f,16,1),P=0;P<t.f;P++)t.e[P]!=t.j[P]&&(e=n.mp(t,P,false),u[P]=t.e[P]==t.j[P]+1&&n.Rm(t,t.k[P][t.j[P]]),t.e[P]=t.j[P],e!=-1&&(D=n.mp(t,P,true),e!=D&&(i=((t.u[P]&bA)>>>27)-1,(i==-1||i<e)&&n.gn(t,P,e))));if(A)for(r=0;r<t.f;r++)u[r]&&n.wq(t,r);t.T=0;},n.rq=function(t,A,e,r){var P,i,D,u;for(n.bw(A,1),i=0;i<e;i++)r[i]!=-1&&t.C[i]!=1&&t.e[i]!=n.Lo(A,r[i])&&(P=n.mp(t,i,false),P!=-1&&(u=n.mp(A,r[i],false),P!=u&&(D=n.zl(A,r[i]),(D==-1||D<P)&&n.gn(A,r[i],P))));},n.sq=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b,g;for(u=0;u<e;u++)if(r[u]!=-1&&t.e[u]>3&&(t.u[u]&O0)!=0){for(g=0,T=-1,y=-1,U=0;U<t.e[u];U++)P[t.k[u][U]]!=-1?++g:t.n[u][U]==1&&n.Rm(t,t.k[u][U])&&t.D[0][t.k[u][U]]==u&&(T=t.k[u][U],y=t.i[u][U]);if(T!=-1&&g>=3){for(i=n.no(t.J[u].a,t.J[u].b,t.J[y].a,t.J[y].b),c=10,b=-1,l=0;l<t.e[u];l++)t.n[u][l]==1&&(!n.Rm(t,t.k[u][l])||t.D[0][t.k[u][l]]==u)&&P[t.k[u][l]]!=-1&&(D=d.Math.abs(n.oo(i,n.Yl(t,u,t.i[u][l]))),c>D&&(c=D,b=t.k[u][l]));b!=-1&&(o=P[b],n.Mn(A,o,t.H[b]==257?129:257),t.D[0][b]!=u&&(n.Dn(A,1,o,r[t.D[0][b]]),n.Dn(A,0,o,r[u])));}}},n.tq=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G;if(t.j[A]!=2||t.n[A][0]!=2||t.n[A][1]!=2||t.j[t.i[A][0]]<2||t.j[t.i[A][1]]<2||t.o[t.i[A][0]]!=1||t.o[t.i[A][1]]!=1){n.un(t,A,0,false);return;}for(O=-1,E=-1,R=-1,k=-1,i=0,y=0;y<2;y++)for(r=t.i[A][y],b=0;b<t.e[r];b++)D=t.i[r][b],D!=A&&(u=t.k[r][b],x=n.Jp(t,u,D),i<x&&(i=x,E=D,O=u,R=r,k=t.i[A][1-y]));if(E!=-1){for(T=0;T<2;T++)for(r=t.i[A][T],b=0;b<t.e[r];b++)D=t.i[r][b],u=t.k[r][b],D!=A&&t.D[0][u]==r&&(t.H[u]=1);for(t.D[1][O]!=E&&(t.D[0][O]=t.D[1][O],t.D[1][O]=E),o=Ot,c=0;c<t.j[R];c++)D=t.i[R][c],D!=A&&o>D&&(o=D);for(F=B(Q.S7,H,6,2,15,1),S=0,U=0;U<t.j[k];U++)D=t.i[k][U],D!=A&&(F[S++]=D);e=n.no(t.J[A].a,t.J[A].b,t.J[k].a,t.J[k].b),P=0,S==2?(F[0]>F[1]&&(G=F[0],F[0]=F[1],F[1]=G),l=n.oo(e,n.Yl(t,k,F[0])),g=n.oo(e,n.Yl(t,k,F[1])),P=l-g):P=n.oo(e,n.Yl(t,k,F[0])),P<0^(t.u[A]&3)==1^o==E?t.H[O]=257:t.H[O]=129;}},n.uq=function(t,A,e,r){var P,i,D,u,o,l;if(P=t.e[A],u=B(Q.S7,H,6,P,15,1),l=n.ep(t,A,e,r,u),l==3)return  false;for(D=(t.u[A]&3)==l?257:129,o=0;o<P;o++)(u[o]&1)==1&&(i=t.k[A][e[o]],t.H[i]=D,t.D[0][i]!=A&&(t.D[1][i]=t.D[0][i],t.D[0][i]=A));return  true;},n.vq=function(t,A){t.T|=248&(8|A);},n.wq=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k;if(n.Bo(t,A),!((t.u[A]&3)==0||(t.u[A]&3)==3)){if(t.o[A]==2&&t.j[A]==2){n.tq(t,A);return;}if(t.j[A]<3||t.j[A]>4){n.un(t,A,0,false);return;}for(e=t.e[A],b=false,D=0;D<e;D++)if(n.fm(t,t.k[A][D])==1){b=true;break;}if(b){for(g=n.Hp(t,A),r=B(Q.Q7,jt,6,e,15,1),u=0;u<e;u++)r[u]=n.Yl(t,t.i[A][g[u]],A);for(o=0;o<e;o++)t.D[0][t.k[A][o]]==A&&n.fm(t,t.k[A][o])==1&&(t.H[t.k[A][o]]=1);if(!((t.p&&A<t.f?n.dt(t.p,A):0)<=24&&n.uq(t,A,g,r))&&(T=n.pq(t,A,true),T!=-1)){for(t.D[0][T]!=A&&(t.D[1][T]=t.D[0][T],t.D[0][T]=A),c=-1,l=0;l<e;l++)if(T==t.k[A][g[l]]){c=l;break;}for(k=j(v(Q.S7,2),E1,7,0,[j(v(Q.S7,1),H,6,15,[2,1,2,1]),j(v(Q.S7,1),H,6,15,[1,2,2,1]),j(v(Q.S7,1),H,6,15,[1,1,2,2]),j(v(Q.S7,1),H,6,15,[2,1,1,2]),j(v(Q.S7,1),H,6,15,[2,2,1,1]),j(v(Q.S7,1),H,6,15,[1,2,1,2])]),i=1;i<e;i++)r[i]<r[0]&&(r[i]+=N1);if(e==3){switch(U=false,c){case 0:U=r[1]<r[2]&&r[2]-r[1]<t1||r[1]>r[2]&&r[1]-r[2]>t1;break;case 1:U=r[2]-r[0]>t1;break;case 2:U=r[1]-r[0]<t1;}P=(t.u[A]&3)==1^U?257:129;}else y=0,r[1]<=r[2]&&r[2]<=r[3]?y=0:r[1]<=r[3]&&r[3]<=r[2]?y=1:r[2]<=r[1]&&r[1]<=r[3]?y=2:r[2]<=r[3]&&r[3]<=r[1]?y=3:r[3]<=r[1]&&r[1]<=r[2]?y=4:r[3]<=r[2]&&r[2]<=r[1]&&(y=5),P=(t.u[A]&3)==1^k[y][c]==1?129:257;t.H[T]=P;}}}},n.xq=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G;if(!((t.F[A]&3)==0||(t.F[A]&3)==3||!n.Vp(t,A))){for(O=-1,R=-1,E=-1,S=-1,P=0,y=0;y<2;y++)for(r=t.D[y][A],b=0;b<t.e[r];b++)u=t.k[r][b],u!=A&&n.fm(t,u)==1&&(D=t.i[r][b],x=n.Jp(t,u,D),P<x&&(P=x,R=D,O=u,E=r,S=t.D[1-y][A]));if(R!=-1){for(T=0;T<2;T++)for(b=0;b<t.e[t.D[T][A]];b++)u=t.k[t.D[T][A]][b],u!=A&&n.fm(t,u)==1&&(t.H[u]=1);for(t.D[1][O]!=R&&(t.D[0][O]=t.D[1][O],t.D[1][O]=R),o=Ot,c=0;c<t.j[E];c++)D=t.i[E][c],t.k[E][c]!=A&&o>D&&(o=D);for(k=B(Q.S7,H,6,2,15,1),F=0,U=0;U<t.j[S];U++)t.k[S][U]!=A&&(k[F++]=t.i[S][U]);i=n.no(t.J[E].a,t.J[E].b,t.J[S].a,t.J[S].b),e=0,F==2?(k[0]>k[1]&&(G=k[0],k[0]=k[1],k[1]=G),l=n.oo(i,n.Yl(t,S,k[0])),g=n.oo(i,n.Yl(t,S,k[1])),e=l-g):e=n.oo(i,n.Yl(t,S,k[0])),e<0^(t.F[A]&3)==2^o==R?t.H[O]=257:t.H[O]=129;}}},n.yq=function(t){var A,e,r,P;for(n.bw(t,7),r=0;r<t.g;r++)(t.H[r]==257||t.H[r]==129)&&(t.H[r]=1);for(A=0;A<t.f;A++)n.wq(t,A);for(P=0;P<t.g;P++)n.xq(t,P);for(e=0;e<t.g;e++)t.H[e]==2&&(t.F[e]&3)==3&&(t.H[e]=386);},n.zq=function(t,A){var e,r,P,i,D,u,o,l,U,y;if(u=B(Q.S7,H,6,t.q,15,1),D=n.hp(t,u,false,A),D<=1)return null;for(o=B(Q.S7,H,6,D,15,1),r=0;r<t.f;r++)++o[u[r]];for(U=0,y=o[0],l=1;l<D;l++)y<o[l]&&(y=o[l],U=l);for(e=0;e<t.q;e++)u[e]!=U&&(t.C[e]=-1);for(i=0;i<t.r;i++)(!A&&t.H[i]==16||u[t.D[0][i]]!=U)&&(t.H[i]=512);P=n.kl(t),t.T=0;try{n.zo(t,!0,!0);}catch(T){if(T=M1(T),!Zt(T,19))throw Lt(T);}return P;},n.Aq=function(t,A){return (t.u[A]&bA)!=0?true:t.C[A]<=1?false:n.Nm(t,A)||t.C[A]==13||t.C[A]>=171;},n.Bq=function(t){var A,e,r,P,i,D,u,o,l,U;for(i=n.Wl(t,t.q,t.r,n.Qk),D=i*i/16,r=1;r<t.q;r++)for(P=0;P<r;P++)if(o=t.J[P].a-t.J[r].a,l=t.J[P].b-t.J[r].b,U=t.J[P].c-t.J[r].c,o*o+l*l+U*U<D)throw Lt(new h.cC("The distance between two atoms is too close."));for(n.bw(t,1),A=0,e=0;e<t.f;e++){if(n.vp(t,e)>(u=n.pm(t,e),u+n.nm(t,e,u,t.C[e])))throw Lt(new h.cC("atom valence exceeded"));A+=t.s[e];}if(A!=0)throw Lt(new h.cC("unbalanced atom charge"));},n.Cq=function(t){var A,e;if(t.K)for(n.bw(t,7),A=0;A<t.f;A++)(t.u[A]&x0)!=0&&(S1(z(t.B[A],120),8)&&(t.B[A]=z(t.B[A],-9)),t.B[A]=z(t.B[A],-4294967297)),A<t.f&&n.ot(t.p,A)?t.B[A]=z(t.B[A],{l:4194297,m:W1,h:1048571}):Rt(z(t.B[A],2),0)&&(t.B[A]=z(t.B[A],-5)),(Rt(z(t.B[A],FA),0)||Rt(z(t.B[A],EA),0)&&S1(z(t.B[A],lA),0))&&S1(z(t.B[A],120),8)&&(t.B[A]=z(t.B[A],-9)),t.s[A]!=0&&(t.B[A]=z(t.B[A],-234881025)),n.vp(t,A)==(e=n.pm(t,A),e+n.nm(t,A,e,t.C[A]))&&(t.B[A]=z(t.B[A],-4063233),t.B[A]=z(t.B[A],-17042430230529),t.B[A]=z(t.B[A],-1921),t.B[A]=z(t.B[A],-114689)),Rt(z(t.B[A],i0),0)&&t.w[A]!=0&&n.an(t,t.w[A]);},n.Dq=function(t){var A,e,r;if(t.K)for(n.bw(t,7),A=0;A<t.g;A++)r=rn(t.G[A]&127),(A<t.g&&(n.ut(t.p,A)||t.H[A]==8))&(t.G[A]&8)!=0&&(t.G[A]&=-9,--r),r!=0&&(e=t.H[A]&127,e==1?t.G[A]|=1:e==2?t.G[A]|=2:e==4?t.G[A]|=4:e==32?t.G[A]|=32:e==64?t.G[A]|=64:e==16?t.G[A]|=16:e==8&&(t.G[A]|=8));},n.Eq=function(t,A,e){var r;return r=n.ao(t,A,e),r&&e==386&&(n.bw(t,7),r=r&(t.F[A]&128)==0),r;},n.Fq=function(){n.eo.call(this);},n.Gq=function(t,A){n.fo.call(this,t,A);},n.Hq=function(t){n.fo.call(this,t?t.M:256,t?t.N:256),t&&n.ol(t,this);},I(114,88,{114:1,88:1,4:1}),L.$b=function(t){n.Eo(this,t);},L.Zb=function(t){return n.Yp(this,t);},L.f=0,L.g=0,Q.r8=N(114),n.Kq=function(t,A){return d.Math.pow(10,d.Math.log(2e3)*d.Math.LOG10E*t/(A-1)-1);},n.Lq=function(t,A){var e,r;for(e=A,r=0;A!=0;)t.d==0&&(t.f=(t.c[++t.e]&63)<<11,t.d=6),r|=(H1&t.f)>>16-e+A,t.f<<=1,--A,--t.d;return r;},n.Mq=function(t,A,e){t.d=6,t.e=e,t.c=A,t.f=(A[t.e]&63)<<11;},n.Nq=function(t,A){var e,r,P,i;return r=A/2|0,P=t>=r,P&&(t-=r),i=A/32,e=i*t/(r-t),P?-e:e;},n.Oq=function(t){var A,e,r,P;if(t.g.K)for(A=0;A<t.g.r;A++)r=n.hm(t.g,A),(r&127)!=0&&(e=-1,P=0,(r&16)!=0&&(e=16,++P),(r&64)!=0&&(e=64,++P),(r&32)!=0&&(e=32,++P),(r&4)!=0&&(e=4,++P),(r&2)!=0&&(e=2,++P),(r&8)!=0&&(e=8,++P),(r&1)!=0&&(e=1,++P),e!=-1&&(n.Mn(t.g,A,e),P==1&&n.Ln(t.g,A,127,false)));},n.Pq=function(t,A){var e;return A==null||w.UAb(A).length==0?null:n.Rq(t,w.uAb((e=A,O1.Enb(),e)));},n.Qq=function(t,A,e){var r,P;return A==null?null:n.Sq(t,w.uAb((P=A,O1.Enb(),P)),e==null?null:w.uAb((r=e,r)),0);},n.Rq=function(t,A){var e;if(A==null||A.length==0)return null;for(e=2;e<A.length-2;e++)if(A[e]==32)return n.Sq(t,A,A,e+1);return n.Sq(t,A,null,0);},n.Sq=function(t,A,e,r){var P,i,D,u,o;return A==null?null:(n.Mq(t,A,0),P=n.Lq(t,4),u=n.Lq(t,4),P>8&&(P=u),i=n.Lq(t,P),D=n.Lq(t,u),o=new n.uw(i,D),n.Vq(t,o,A,e,r),o);},n.Tq=function(t,A,e){var r,P,i,D;if(e==null||w.UAb(e).length==0){n.Uq(t,A,null,null);return;}r=h.jkb(e,X1(32)),r>0&&r<w.UAb(e).length-1?n.Uq(t,A,w.uAb((i=(w.OAb(0,r,w.UAb(e).length),w.UAb(e).substr(0,r)),O1.Enb(),i)),w.uAb((D=(w.PAb(r+1,w.UAb(e).length+1),w.UAb(e).substr(r+1)),D))):n.Uq(t,A,w.uAb((P=e,O1.Enb(),P)),null);},n.Uq=function(t,A,e,r){if(e==null||e.length==0){n.jl(A),A.d=null;return;}n.Vq(t,A,e,r,0);},n.Vq=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt,Gt,dt,Bt,b1,f1,St,gt,g1,q1,U1,G1,y1,o1,K1,t0,R0,z1,u1,u0,h0,d0,CA,BA,nA,eA,C0,MA,J0,M0,YA,dn,vt,Wt,kA,mn,Fn,vn,i8,f8,D8,u8,o8,Q8,l8,U8,Or,xr,L8,s8,y8,T8,h8,c8,S0,b8,w8,g8,B8,k8,d8,m8,F8,v8,j8,oA,R8,Pn,p8,_n,ru,Pu,iu,fu,Du,$t,C8,uu,ou,Qu,XA,lu,QA,Wn,qr,Ce,Uu,rA,Vr,Lu,Me,su,Se,yu,He;if(n.jl(A),A.d=null,!(e==null||0>=e.length)){if(t.g=A,Vr=8,r!=null&&(P<0||P>=r.length)&&(r=null),n.Mq(t,e,0),u=n.Lq(t,4),G=n.Lq(t,4),u>8&&(Vr=u,u=G),u==0){n.On(t.g,n.Lq(t,1)==1);return;}for(o=n.Lq(t,u),l=n.Lq(t,G),Du=n.Lq(t,u),Qu=n.Lq(t,u),uu=n.Lq(t,u),st=n.Lq(t,u),c=0;c<o;c++)n.Uk(t.g,6);for(h0=0;h0<Du;h0++)n.Cn(t.g,n.Lq(t,u),7);for(d0=0;d0<Qu;d0++)n.Cn(t.g,n.Lq(t,u),8);for(vt=0;vt<uu;vt++)n.Cn(t.g,n.Lq(t,u),n.Lq(t,8));for(Q8=0;Q8<st;Q8++)n.jn(t.g,n.Lq(t,u),n.Lq(t,4)-8);for(wt=1+l-o,dt=n.Lq(t,4),x=0,n.zn(t.g,0,0),n.An(t.g,0,0),n.Bn(t.g,0,0),Bt=r!=null&&r[P]>=39,rA=0,Me=0,Se=0,He=0,Ht=false,Jt=false,Bt&&(r.length>2*o-2&&r[2*o-2]==39||r.length>3*o-3&&r[3*o-3]==39?(Jt=true,Ht=r.length==3*o-3+9,oA=Ht?3*o-3:2*o-2,O=86*(r[oA+1]-40)+r[oA+2]-40,rA=d.Math.pow(10,O/2e3-1),oA+=2,Lu=86*(r[oA+1]-40)+r[oA+2]-40,Me=d.Math.pow(10,Lu/1500-1),oA+=2,su=86*(r[oA+1]-40)+r[oA+2]-40,Se=d.Math.pow(10,su/1500-1),Ht&&(oA+=2,yu=86*(r[oA+1]-40)+r[oA+2]-40,He=d.Math.pow(10,yu/1500-1))):Ht=r.length==3*o-3),t._b()&&Ht&&(r=null,Bt=false),S0=1;S0<o;S0++){if(St=n.Lq(t,dt),St==0){Bt&&(n.zn(t.g,S0,n.Ql(t.g,0)+8*(r[S0*2-2]-83)),n.An(t.g,S0,n.Rl(t.g,0)+8*(r[S0*2-1]-83)),Ht&&n.Bn(t.g,S0,n.Sl(t.g,0)+8*(r[2*o-3+S0]-83))),++wt;continue;}x+=St-1,Bt&&(n.zn(t.g,S0,n.Ql(t.g,x)+r[S0*2-2]-83),n.An(t.g,S0,n.Rl(t.g,x)+r[S0*2-1]-83),Ht&&n.Bn(t.g,S0,n.Sl(t.g,x)+(r[2*o-3+S0]-83))),n.Wk(t.g,x,S0,1);}for(d8=0;d8<wt;d8++)n.Wk(t.g,n.Lq(t,u),n.Lq(t,u),1);for(R8=B(Q.xfb,Tt,6,l,16,1),it=0;it<l;it++)switch(lt=n.Lq(t,2),lt){case 0:R8[it]=true;break;case 2:n.Mn(t.g,it,2);break;case 3:n.Mn(t.g,it,4);}for(D=n.Lq(t,u),m8=0;m8<D;m8++)if(T=n.Lq(t,u),Vr==8)XA=n.Lq(t,2),XA==3?(n.on(t.g,T,1,0),n.un(t.g,T,1,false)):n.un(t.g,T,XA,false);else switch(XA=n.Lq(t,3),XA){case 4:n.un(t.g,T,1,false),n.on(t.g,T,1,n.Lq(t,3));break;case 5:n.un(t.g,T,2,false),n.on(t.g,T,1,n.Lq(t,3));break;case 6:n.un(t.g,T,1,false),n.on(t.g,T,2,n.Lq(t,3));break;case 7:n.un(t.g,T,2,false),n.on(t.g,T,2,n.Lq(t,3));break;default:n.un(t.g,T,XA,false);}for(Vr==8&&n.Lq(t,1)==0&&(t.g.L=true),i=n.Lq(t,G),F8=0;F8<i;F8++)if(V=n.Lq(t,G),n.im(t.g,V)==1)switch(XA=n.Lq(t,3),XA){case 4:n.Jn(t.g,V,1,false),n.Gn(t.g,V,1,n.Lq(t,3));break;case 5:n.Jn(t.g,V,2,false),n.Gn(t.g,V,1,n.Lq(t,3));break;case 6:n.Jn(t.g,V,1,false),n.Gn(t.g,V,2,n.Lq(t,3));break;case 7:n.Jn(t.g,V,2,false),n.Gn(t.g,V,2,n.Lq(t,3));break;default:n.Jn(t.g,V,XA,false);}else n.Jn(t.g,V,n.Lq(t,2),false);for(n.On(t.g,n.Lq(t,1)==1),y=null,Ce=null,C8=0;n.Lq(t,1)==1;)switch(Gt=C8+n.Lq(t,4),Gt){case 0:for($t=n.Lq(t,u),v8=0;v8<$t;v8++)T=n.Lq(t,u),n.vn(t.g,T,B1,true);break;case 1:for($t=n.Lq(t,u),j8=0;j8<$t;j8++)T=n.Lq(t,u),iu=n.Lq(t,8),n.tn(t.g,T,iu);break;case 2:for($t=n.Lq(t,G),CA=0;CA<$t;CA++)n.Lq(t,G);break;case 3:for($t=n.Lq(t,u),BA=0;BA<$t;BA++)T=n.Lq(t,u),n.vn(t.g,T,Z1,true);break;case 4:for($t=n.Lq(t,u),nA=0;nA<$t;nA++)T=n.Lq(t,u),qr=c1(Mt(n.Lq(t,4)),3),n.vn(t.g,T,qr,true);break;case 5:for($t=n.Lq(t,u),eA=0;eA<$t;eA++)T=n.Lq(t,u),U=c1(Mt(n.Lq(t,2)),1),n.vn(t.g,T,U,true);break;case 6:for($t=n.Lq(t,u),C0=0;C0<$t;C0++)T=n.Lq(t,u),n.vn(t.g,T,1,true);break;case 7:for($t=n.Lq(t,u),MA=0;MA<$t;MA++)T=n.Lq(t,u),z1=c1(Mt(n.Lq(t,4)),7),n.vn(t.g,T,z1,true);break;case 8:for($t=n.Lq(t,u),J0=0;J0<$t;J0++){for(T=n.Lq(t,u),S=n.Lq(t,4),k=B(Q.S7,H,6,S,15,1),Pn=0;Pn<S;Pn++)F=n.Lq(t,8),k[Pn]=F;n.pn(t.g,T,k);}break;case 9:for($t=n.Lq(t,G),M0=0;M0<$t;M0++)V=n.Lq(t,G),qr=n.Lq(t,2)<<7,n.Ln(t.g,V,qr,true);break;case 10:for($t=n.Lq(t,G),YA=0;YA<$t;YA++)V=n.Lq(t,G),Dt=n.Lq(t,5),n.Ln(t.g,V,Dt,true);break;case 11:for($t=n.Lq(t,u),dn=0;dn<$t;dn++)T=n.Lq(t,u),n.vn(t.g,T,U0,true);break;case 12:for($t=n.Lq(t,G),Wt=0;Wt<$t;Wt++)V=n.Lq(t,G),Qt=n.Lq(t,8)<<9,n.Ln(t.g,V,Qt,true);break;case 13:for($t=n.Lq(t,u),kA=0;kA<$t;kA++)T=n.Lq(t,u),lu=c1(Mt(n.Lq(t,3)),14),n.vn(t.g,T,lu,true);break;case 14:for($t=n.Lq(t,u),mn=0;mn<$t;mn++)T=n.Lq(t,u),fu=c1(Mt(n.Lq(t,5)),17),n.vn(t.g,T,fu,true);break;case 15:case 31:C8+=16;break;case 16:for($t=n.Lq(t,u),Fn=0;Fn<$t;Fn++)T=n.Lq(t,u),Wn=c1(Mt(n.Lq(t,3)),22),n.vn(t.g,T,Wn,true);break;case 17:for($t=n.Lq(t,u),vn=0;vn<$t;vn++)T=n.Lq(t,u),n.gn(t.g,T,n.Lq(t,4));break;case 18:for($t=n.Lq(t,u),Pu=n.Lq(t,4),i8=0;i8<$t;i8++){for(T=n.Lq(t,u),Nt=n.Lq(t,Pu),p8=B(Q.O7,Y,6,Nt,15,1),Pn=0;Pn<Nt;Pn++)p8[Pn]=n.Lq(t,7)<<24>>24;n.mn(t.g,T,h.akb((_n=p8,O1.Enb(),_n)));}break;case 19:for($t=n.Lq(t,u),f8=0;f8<$t;f8++)T=n.Lq(t,u),bt=c1(Mt(n.Lq(t,3)),25),n.vn(t.g,T,bt,true);break;case 20:for($t=n.Lq(t,G),D8=0;D8<$t;D8++)V=n.Lq(t,G),Wn=n.Lq(t,3)<<17,n.Ln(t.g,V,Wn,true);break;case 21:for($t=n.Lq(t,u),u8=0;u8<$t;u8++)T=n.Lq(t,u),n.wn(t.g,T,n.Lq(t,2)<<4);break;case 22:for($t=n.Lq(t,u),o8=0;o8<$t;o8++)T=n.Lq(t,u),n.vn(t.g,T,NA,true);break;case 23:for($t=n.Lq(t,G),l8=0;l8<$t;l8++)V=n.Lq(t,G),n.Ln(t.g,V,TA,true);break;case 24:for($t=n.Lq(t,G),U8=0;U8<$t;U8++)V=n.Lq(t,G),U=n.Lq(t,2)<<21,n.Ln(t.g,V,U,true);break;case 25:for(Or=0;Or<o;Or++)n.Lq(t,1)==1&&n.xn(t.g,Or,true);break;case 26:for($t=n.Lq(t,G),y=B(Q.S7,H,6,$t,15,1),xr=0;xr<$t;xr++)y[xr]=n.Lq(t,G);break;case 27:for($t=n.Lq(t,u),L8=0;L8<$t;L8++)T=n.Lq(t,u),n.vn(t.g,T,i0,true);break;case 28:for($t=n.Lq(t,G),s8=0;s8<$t;s8++)n.Mn(t.g,n.Lq(t,G),16);break;case 29:for($t=n.Lq(t,u),y8=0;y8<$t;y8++)T=n.Lq(t,u),R0=c1(Mt(n.Lq(t,2)),30),n.vn(t.g,T,R0,true);break;case 30:for($t=n.Lq(t,u),T8=0;T8<$t;T8++)T=n.Lq(t,u),Wn=c1(Mt(n.Lq(t,7)),32),n.vn(t.g,T,Wn,true);break;case 32:for($t=n.Lq(t,u),h8=0;h8<$t;h8++)T=n.Lq(t,u),Uu=c1(Mt(n.Lq(t,2)),44),n.vn(t.g,T,Uu,true);break;case 33:for($t=n.Lq(t,u),c8=0;c8<$t;c8++)T=n.Lq(t,u),G1=c1(Mt(n.Lq(t,5)),39),n.vn(t.g,T,G1,true);break;case 34:for($t=n.Lq(t,u),b8=0;b8<$t;b8++)T=n.Lq(t,u),n.vn(t.g,T,ee,true);break;case 35:for($t=n.Lq(t,G),w8=0;w8<$t;w8++)V=n.Lq(t,G),n.Ln(t.g,V,Ln,true);break;case 36:for($t=n.Lq(t,G),g8=0;g8<$t;g8++)V=n.Lq(t,G),ut=n.Lq(t,2)<<5,n.Ln(t.g,V,ut,true);break;case 37:for($t=n.Lq(t,G),B8=0;B8<$t;B8++)V=n.Lq(t,G),ut=n.Lq(t,1)==0?32:64,n.Mn(t.g,V,ut);break;case 38:for($t=n.Lq(t,u),ct=n.Lq(t,3),Ce=B(Q.S7,H,6,o,15,1),k8=0;k8<$t;k8++)T=n.Lq(t,u),Vt=n.Lq(t,ct),Ce[T]=Vt;break;case 39:for($t=n.Lq(t,u),u0=0;u0<$t;u0++)T=n.Lq(t,u),ou=c1(Mt(n.Lq(t,4)),47),n.vn(t.g,T,ou,true);}if(n.Ge(new n.Ve(t.g),R8,false),y!=null)for(At=y,ot=0,K=At.length;ot<K;++ot)V=At[ot],n.Mn(t.g,V,n.im(t.g,V)==2?4:2);if(r==null&&!t.i&&e.length>t.e+1&&(e[t.e+1]==32||e[t.e+1]==9)&&(r=e,P=t.e+2),r!=null)try{if(r[P]==33||r[P]==35){for(n.Mq(t,r,P+1),Ht=n.Lq(t,1)==1,Jt=n.Lq(t,1)==1,QA=2*n.Lq(t,4),J=1<<QA,K1=0,V=0,b=1;b<o;b++)V<l&&n.Zl(t.g,1,V)==b?(K1=n.Zl(t.g,0,V++),o1=1):(K1=0,o1=8),b1=o1*(n.Lq(t,QA)+1-(J>>1)),f1=o1*(n.Lq(t,QA)+1-(J>>1)),n.zn(t.g,b,n.Ql(t.g,K1)+b1),n.An(t.g,b,n.Rl(t.g,K1)+f1),Ht&&n.Bn(t.g,b,n.Sl(t.g,K1)+o1*(n.Lq(t,QA)+1-(J>>1)));if(r[P]==35){for(u1=0,t0=B(Q.S7,H,6,o,15,1),g=0;g<o;g++)u1+=t0[g]=n.op(t.g,g);for(T=0;T<o;T++)for(u0=0;u0<t0[T];u0++)z1=n.Uk(t.g,1),n.Wk(t.g,T,z1,1),n.zn(t.g,z1,n.Ql(t.g,T)+(n.Lq(t,QA)+1-(J>>1))),n.An(t.g,z1,n.Rl(t.g,T)+(n.Lq(t,QA)+1-(J>>1))),Ht&&n.Bn(t.g,z1,n.Sl(t.g,T)+(n.Lq(t,QA)+1-(J>>1))),Ce!=null&&(Ce[T]&1<<u0)!=0&&n.xn(t.g,z1,!0);o+=u1,l+=u1;}if(E=Ht?1.5:(n.Rk(),n.Rk(),n.Qk),R=n.Wl(t.g,o,l,E),Jt)for(rA=n.Kq(n.Lq(t,QA),J),Me=rA*n.Nq(n.Lq(t,QA),J),Se=rA*n.Nq(n.Lq(t,QA),J),Ht&&(He=rA*n.Nq(n.Lq(t,QA),J)),o1=rA/R,T=0;T<o;T++)n.zn(t.g,T,Me+o1*n.Ql(t.g,T)),n.An(t.g,T,Se+o1*n.Rl(t.g,T)),Ht&&n.Bn(t.g,T,He+o1*n.Sl(t.g,T));else for(rA=1.5,o1=rA/R,T=0;T<o;T++)n.zn(t.g,T,o1*n.Ql(t.g,T)),n.An(t.g,T,o1*n.Rl(t.g,T)),Ht&&n.Bn(t.g,T,o1*n.Sl(t.g,T));}else if(Ht&&!Jt&&rA==0&&(rA=1.5),rA!=0&&t.g.r!=0){for(R=0,V=0;V<t.g.r;V++)gt=n.Ql(t.g,n.Zl(t.g,0,V))-n.Ql(t.g,n.Zl(t.g,1,V)),g1=n.Rl(t.g,n.Zl(t.g,0,V))-n.Rl(t.g,n.Zl(t.g,1,V)),q1=Ht?n.Sl(t.g,n.Zl(t.g,0,V))-n.Sl(t.g,n.Zl(t.g,1,V)):0,R+=d.Math.sqrt(gt*gt+g1*g1+q1*q1);for(R/=t.g.r,y1=rA/R,T=0;T<t.g.q;T++)n.zn(t.g,T,n.Ql(t.g,T)*y1+Me),n.An(t.g,T,n.Rl(t.g,T)*y1+Se),Ht&&n.Bn(t.g,T,n.Sl(t.g,T)*y1+He);}}catch(dA){if(dA=M1(dA),Zt(dA,19))U1=dA,h.MB(U1,(h.Rkb(),h.Qkb),""),""+U1+h.akb((ru=e,O1.Enb(),ru))+h.akb((_n=r,_n)),r=null,Ht=false;else throw Lt(dA);}if(Et=r!=null&&!Ht,n.Oq(t),Et||t._b())for(n.bw(t.g,7),V=0;V<t.g.g;V++)n.fm(t.g,V)==2&&!n.jq(t.g,V)&&n.gm(t.g,V)==0&&n.Kn(t.g,V);if(n.vq(t.g,0),!Et&&t._b())try{t.ac(t.g),Et=!0;}catch(dA){if(dA=M1(dA),Zt(dA,19))U1=dA,h.MB(U1,(h.Rkb(),h.Qkb),""),""+U1+h.akb((_n=e,O1.Enb(),_n));else throw Lt(dA);}Et?(n.yq(t.g),n.ow(t.g)):Ht||n.vq(t.g,0);}},n.Wq=function(t,A){var e,r,P,i,D,u;if(!(A.length<=0||A[0]<64))for(n.Mq(t,A,0),u=n.Lq(t,4),P=n.Lq(t,1)==1,i=n.Lq(t,1)==1,e=0;e<t.g.f;e++)D=n.Lq(t,u),r=P,P&&i&&(r=n.Lq(t,1)==1),n.rn(t.g,e,D,r);},n.Xq=function(){},I(319,1,{},n.Xq),L._b=function(){return  false;},L.ac=function(t){throw Lt(new h.cC("Unexpected request to invent coordinates. Check source code logic!"));},L.d=0,L.e=0,L.f=0,L.i=false,Q.s8=N(319),n.$q=function(t){},n._q=function(t){n.$q(this),this.b=t;},I(51,319,{},n._q),L._b=function(){return this.b;},L.ac=function(t){var A;A=new X.CA(this.a),A.g=new s.wwb(78187493520),X.jA(A,t);},L.a=2,L.b=false,Q.t8=N(51),n.cr=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k,F;P=A.a,c=A.g,l=Rt(z(n.Ol(t.k,P),1),0),i=n.Jl(t.k,P),y=i!=null?n.jr(t,P,i,r):l?"*":n.Il(t.k,P),!l&&i==null&&n.Tp(t.k,P)&&(t.j&4)==0&&(n.Qo(t.k,P)!=0||n.zl(t.k,P)==-1&&n.Pl(t.k,P)==0)&&(y=w.UAb(y).toLowerCase()),A.f&&(e.a+="("),c!=-1&&n.gr(t,n.Xo(t.k,A.a,A.g),A.g,e),D=n.Bl(t.k,P),D==0&&(t.j&1)!=0&&(u=z(n.Ol(t.k,P),mA),nt(u,Qn)==0?D=-1:nt(u,on)==0&&(D=1)),U=n.Ml(t.k,P),T=(t.j&2)!=0?n.Ll(t.k,P):0,b=(t.j&1)!=0?n.pr(t,P,r):null,F=!l&&!n.wr(n.Tl(t.k,P))||i!=null||(n.Nl(t.k,P)==1||n.Nl(t.k,P)==2)&&(k=n.Hl(t.k,P)-1,!(k!=-1&&t.i[k][n.Gl(t.k,P)]<=1))&&(n.Tl(t.k,P)!=7||n.Bl(t.k,P)>0)||n.Tp(t.k,P)&&n.Qo(t.k,P)==0&&(t.j&4)==0||D!=0||U!=0||T!=0||n.zl(t.k,P)!=-1||n.Pl(t.k,P)!=0||b!=null,F&&(e.a+="["),U!=0&&(e.a+=U),e.a+=""+y,(n.Nl(t.k,P)==1||n.Nl(t.k,P)==2)&&(g=n.Hl(t.k,P)-1,!(g!=-1&&t.i[g][n.Gl(t.k,P)]<=1))&&(n.Tl(t.k,P)!=7||n.Bl(t.k,P)>0)&&h.Kkb(e,n.or(t,P,c)),(t.j&1)==0&&F&&(o=n.Cp(t.k,P),o==1?e.a+="H":o>1&&(e.a+="H"+o)),D!=0&&(e.a+=String.fromCharCode(D>0?43:45),d.Math.abs(D)>1&&h.Hkb(e,d.Math.abs(D))),b!=null&&(e.a+=""+b),T!=0&&(e.a+=":",e.a+=T),F&&(e.a+="]"),n.hr(t,A,e),A.e&&(e.a+=")");},n.dr=function(t,A,e){var r,P,i,D,u;for(i=true,u=s.xj(t.g,A).a,D=n.sr(t,u);D!=-1;)r=n.Zo(t.k,u,D),P=n._o(t.k,u,D),D=n.sr(t,r),n.fr(t,new n.$u(r,P,u,e&&i,e&&D==-1),++A),u=r,i=false;},n.er=function(t,A,e){var r,P,i,D;for(P=A,i=0,D=P.length;i<D;++i)r=P[i],t.f[r&1073741823]=(r&q0)!=0^e?2:1;},n.fr=function(t,A,e){s.rj(t.g,e,A),t.b[A.a]=true,A.g!=-1&&(t.c[n.Xo(t.k,A.a,A.g)]=true);},n.gr=function(t,A,e,r){var P,i,D,u,o;o=w.UAb(r.a).length,t.f[A]!=0&&h.Gkb(r,t.f[A]==1?47:92),t.j==1&&(P=n.hm(t.k,A)&127,P!=0&&((P&1)!=0&&t.f[A]==0&&(r.a+="-"),(P&2)!=0&&(w.UAb(r.a).length!=o&&(r.a+=","),r.a+="="),(P&4)!=0&&(w.UAb(r.a).length!=o&&(r.a+=","),r.a+="#"),(P&32)!=0&&(w.UAb(r.a).length!=o&&(r.a+=","),r.a+="$"),(P&64)!=0&&(w.UAb(r.a).length!=o&&(r.a+=","),r.a+="$"),(P&8)!=0&&(w.UAb(r.a).length!=o&&(r.a+=","),r.a+=":"),(P&16)!=0&&(w.UAb(r.a).length!=o&&(r.a+=","),h.Kkb(r,n.Lm(t.k,e)?"<-":"->")))),o==w.UAb(r.a).length&&(!n.Up(t.k,A)||(t.j&4)!=0)&&(i=n.im(t.k,A)&127,i==1?n.Tp(t.k,n.Zl(t.k,0,A))&&n.Tp(t.k,n.Zl(t.k,1,A))&&(t.j&4)==0&&t.f[A]==0&&(r.a+="-"):i==2?r.a+="=":i==4?r.a+="#":i==32||i==64?r.a+="$":i==8?r.a+=":":i==16&&h.Kkb(r,n.Lm(t.k,e)?"<-":"->")),t.j==1&&(D=o==w.UAb(r.a).length?"":";",u=n.hm(t.k,A)&384,u==256?r.a+=D+"@":u==128&&(r.a+=D+"!@"));},n.hr=function(t,A,e){var r,P,i;if(A.c!=null)for(P=0;P<A.c.length;P++)for(i=0;i<n.$o(t.k,A.a);i++)A.c[P]==n.Zo(t.k,A.a,i)&&(r=n._o(t.k,A.a,i),A.d[P]||n.gr(t,r,A.a,e),t.e[r]>9&&(e.a+="%"),h.Hkb(e,t.e[r]));},n.ir=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt;for(bt=new s.Mj(),F=new s.trb(t.g);F.a<F.c.a.length;)if(k=s.srb(F),k.g!=-1&&(R=n.Xo(t.k,k.a,k.g),!n.Vp(t.k,R)&&!n.jq(t.k,R)&&(n.gm(t.k,R)==1||n.gm(t.k,R)==2))){if(Dt=s.xj(t.g,t.q[k.g]),e=B(Q.S7,H,6,n.$o(t.k,k.a)+n.$o(t.k,Dt.a)-2,15,1),J=0,Qt=false,Dt.g!=-1)e[J++]=Dt.b;else {for(E=-1,st=-1,O=Ot,At=0;At<n.$o(t.k,Dt.a);At++)y=n.Zo(t.k,Dt.a,At),y!=k.a&&(E==-1?(E=At,O=t.q[y]):O<t.q[y]?st=At:(st=E,E=At));st==-1?(it=n.Zo(t.k,Dt.a,E),lt=n._o(t.k,Dt.a,E),e[J++]=lt|(n.ur(t,Dt.a,it)?q0:0)):(T=n.Zo(t.k,Dt.a,E),b=n._o(t.k,Dt.a,E),c=n.Zo(t.k,Dt.a,st),g=n._o(t.k,Dt.a,st),e[J++]=b|(n.ur(t,Dt.a,T)?q0:0),e[J++]=g|(n.ur(t,Dt.a,c)?0:q0));}if(n.$o(t.k,Dt.a)==3&&Dt.g!=-1){for(At=0;At<n.$o(t.k,Dt.a);At++)if(y=n.Zo(t.k,Dt.a,At),y!=Dt.g&&y!=k.a){r=n._o(t.k,Dt.a,At),e[J++]=r|(n.ur(t,Dt.a,y)?q0:0),y<Dt.g&&(Qt=!Qt);break;}}for(n.gm(t.k,R)==2&&(Qt=!Qt),V=0;V<n.$o(t.k,k.a);V++)if(o=n.Zo(t.k,k.a,V),o!=k.g){if(G=Qt,n.$o(t.k,k.a)==3){for(K=0;K<n.$o(t.k,k.a);K++)if(y=n.Zo(t.k,k.a,K),y!=k.g&&y!=o){y<o&&(G=!G);break;}}n.Fm(t.k,R)&&(x=n.tg(t.d,R),t.n[x]||(t.n[x]=true,t.o[x]=G),t.o[x]&&(G=!G)),l=n.Xo(t.k,k.a,o),e[J++]=l|(G^n.ur(t,k.a,o)?0:q0);}w.kAb(bt.a,e);}for(t.f=B(Q.S7,H,6,t.k.g,15,1),bt.a.length!=0&&n.er(t,s.Ej(bt,0),false);bt.a.length!=0;){for(wt=bt.a.length,V=bt.a.length-1;V>=0;V--){for(e=(w.HAb(V,bt.a.length),bt.a[V]),ut=0,ot=false,U=false,i=e,D=0,u=i.length;D<u;++D)P=i[D],A=P&1073741823,t.f[A]!=0&&(S=(P&q0)!=0^t.f[A]==2,ut==0?ot=S:ot!=S&&(U=true),++ut);ut!=0&&(e=s.Ej(bt,V),U||n.er(t,e,ot));}wt==bt.a.length&&n.er(t,s.Ej(bt,0),false);}},n.jr=function(t,A,e,r){var P,i,D,u,o,l,U;for(U=w.UAb(r.a).length,0<U?r.a=h.tkb(r.a,0,0):0>U&&(r.a+=h.zkb(B(Q.P7,e0,6,-U,15,1))),o=n.Tp(t.k,A)&&(t.j&4)==0,i=e,D=0,u=i.length;D<u;++D)P=i[D],w.UAb(r.a).length!=0&&(r.a+=","),l=(n.Rk(),n.Lk)[P],h.Kkb(r,o?w.UAb(l).toLowerCase():l);return r.a;},n.kr=function(t){var A,e,r,P,i,D,u,o;if(!t.k||t.k.q==0)return "";for(n.bw(t.k,15),t.d=new n.Og(t.k,129),P=t.d.T,t.o=B(Q.xfb,Tt,6,P+1,16,1),t.n=B(Q.xfb,Tt,6,P+1,16,1),t.i=T1(Q.S7,[E1,H],[7,6],15,[2,32],2),A=0;A<t.k.f;A++)o=n.Hl(t.k,A)-1,o!=-1&&++t.i[o][n.Gl(t.k,A)];for(n.nr(t),n.lr(t),n.ir(t),r=new h.Mkb(),e=new h.Mkb(),i=true,u=new s.trb(t.g);u.a<u.c.a.length;)D=s.srb(u),D.g==-1&&(i?i=false:r.a+="."),n.cr(t,D,r,e);return r.a;},n.lr=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(y=new s.trb(t.g);y.a<y.c.a.length;){for(l=s.srb(y),e=0,i=0;i<n.$o(t.k,l.a);i++)t.c[n._o(t.k,l.a,i)]||++e;if(e!=0){for(l.c=B(Q.S7,H,6,e,15,1),e=0,D=0;D<n.$o(t.k,l.a);D++)t.c[n._o(t.k,l.a,D)]||(o=n.Zo(t.k,l.a,D),l.c[e++]=t.q[o]<<16|o);for(w.nAb(l.c,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),P=0;P<l.c.length;P++)l.c[P]=n1&l.c[P];}}for(r=B(Q.xfb,Tt,6,t.k.g,16,1),t.e=B(Q.S7,H,6,t.k.g,15,1),U=new s.trb(t.g);U.a<U.c.a.length;)if(l=s.srb(U),l.c!=null){for(l.d=B(Q.xfb,Tt,6,l.c.length,16,1),P=0;P<l.c.length;P++)for(u=0;u<n.$o(t.k,l.a);u++)if(l.c[P]==n.Zo(t.k,l.a,u))if(A=n._o(t.k,l.a,u),t.c[A])r[t.e[A]]=false;else {for(t.c[A]=true,l.d[P]=true,t.e[A]=1;r[t.e[A]];)++t.e[A];r[t.e[A]]=true;}}},n.mr=function(t){var A,e,r,P;for(r=-1,P=Ot,A=0;A<t.k.f;A++)t.b[A]||(e=t.a[A],n.Jm(t.k,A)&&(e+=q0),n.$o(t.k,A)==0?e+=1056964608:e+=n.$o(t.k,A)<<24,P>e&&(P=e,r=A));return r;},n.nr=function(t){var A,e,r,P,i;for(t.a=t.d.f,t.b=B(Q.xfb,Tt,6,t.k.f,16,1),t.c=B(Q.xfb,Tt,6,t.k.g,16,1),t.g=new s.Mj(),A=n.mr(t);A!=-1;){if(e=t.g.a.length,n.fr(t,new n.$u(A,-1,-1,false,false),e),n.$o(t.k,A)!=0)for(n.dr(t,e,false);e<t.g.a.length-1;){for(;n.tr(t,s.xj(t.g,e).a);)n.dr(t,e,true);++e;}A=n.mr(t);}for(t.q=B(Q.S7,H,6,t.k.f,15,1),r=0,i=new s.trb(t.g);i.a<i.c.a.length;)P=s.srb(i),t.q[P.a]=r++;},n.or=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b;if(o=false,n.Qo(t.k,A)!=0&&n.$o(t.k,A)==2&&n.ap(t.k,A,0)==2&&n.ap(t.k,A,1)==2){for(D=0;D<n.$o(t.k,A);D++)if(r=n.Zo(t.k,A,D),r!=e&&n.$o(t.k,r)==2){o=true;break;}for(i=0;i<n.$o(t.k,A);i++){for(r=n.Zo(t.k,A,i),b=0,c=B(Q.S7,H,6,3,15,1),U=0;U<n.$o(t.k,r);U++)c[b]=n.Zo(t.k,r,U),c[b]!=A&&++b;b==2&&t.q[c[0]]<t.q[c[1]]^c[0]<c[1]&&(o=!o);}}else {for(y=B(Q.S7,H,6,4,15,1),T=B(Q.S7,H,6,4,15,1),u=0,e!=-1&&(y[u]=e,T[u++]=8*t.q[e]),(n.op(t.k,A)!=0||n.$o(t.k,A)==3)&&(y[u]=Ot,T[u++]=8*t.q[A]),i=0;i<n.$o(t.k,A);i++)r=n.Zo(t.k,A,i),r!=e&&(y[u]=r,T[u++]=n.rr(t,A,i));o=n.vr(y,T);}return l=n.Nl(t.k,A)==1^o,n.ym(t.k,A)&&(P=n.ug(t.d,A),t.n[P]||(t.n[P]=true,t.o[P]=l),t.o[P]&&(l=!l)),l?"@@":"@";},n.pr=function(t,A,e){var r,P,i,D,u,o,l,U;switch(u=w.UAb(e.a).length,0<u?e.a=h.tkb(e.a,0,0):0>u&&(e.a+=h.zkb(B(Q.P7,e0,6,-u,15,1))),o=n.Ol(t.k,A),P=zt(A0(z(o,mA),3)),P){case 20971520:e.a+="+0";break;case 25165824:n.Bl(t.k,A)==0&&(e.a+="-");break;case 12582912:n.Bl(t.k,A)==0&&(e.a+="+");}return r=z(o,Xe),nt(r,2)==0?e.a+=";a":nt(r,4)==0&&(e.a+=";A"),i=z(o,X0),nt(i,0)!=0&&(nt(i,1792)==0?e.a+=";H0":nt(i,1664)==0?e.a+=";H1":nt(i,1408)==0?e.a+=";H2":nt(i,896)==0?e.a+=";H3":nt(i,128)==0?e.a+=";!H0":nt(i,384)==0?e.a+=";!H0;!H1":nt(i,1536)==0?e.a+=";!H2;!H3":nt(i,f0)==0&&(e.a+=";!H3")),U=z(o,120),nt(U,8)==0?e.a+=";!R0":nt(U,16)==0?e.a+=";!R1":nt(U,32)==0?e.a+=";!R2":nt(U,64)==0?e.a+=";!R3":nt(U,112)==0?e.a+=";R0":nt(U,104)==0?e.a+=";R1":nt(U,88)==0?e.a+=";R2":nt(U,56)==0&&(e.a+=";R3"),l=z(o,EA),nt(l,lA)==0?e.a+=";!r"+s0(l):nt(l,J8)==0?e.a+=";r"+s0(l):nt(l,0)!=0&&(Rt(z(l,un),0)?(S1(z(l,lA),0)&&(e.a+=";!r0"+s0(l)),S1(z(l,OA),0)&&(e.a+=";!r3"+s0(l)),S1(z(l,xA),0)&&(e.a+=";!r4"+s0(l)),S1(z(l,qA),0)&&(e.a+=";!r5"+s0(l)),S1(z(l,VA),0)&&(e.a+=";!r6"+s0(l)),S1(z(l,IA),0)&&(e.a+=";!r7"+s0(l))):(e.a+=";",Rt(z(l,lA),0)&&(e.a+="r0,"+s0(l)),Rt(z(l,OA),0)&&(e.a+="r3,"+s0(l)),Rt(z(l,xA),0)&&(e.a+="r4,"+s0(l)),Rt(z(l,qA),0)&&(e.a+="r5,"+s0(l)),Rt(z(l,VA),0)&&(e.a+="r6,"+s0(l)),Rt(z(l,IA),0)&&(e.a+="r7,"+s0(l)),h.yhb(e,w.UAb(e.a).length-1))),nt(l,0)==0&&(l=A0(z(o,FA),22),nt(l,0)!=0&&(e.a+=";r"+s0(l))),D=z(o,aA),nt(D,Pe)==0&&(e.a+=";D1"),nt(D,pn)==0&&(e.a+=";D2"),nt(D,Cn)==0&&(e.a+=";D3"),nt(D,ie)==0&&(e.a+=";!D3;!D4"),nt(D,j0)==0&&(e.a+=";!D4"),nt(D,tn)==0&&(e.a+=";!D0;!D1"),nt(D,iA)==0&&(e.a+=";!D0;!D1;!D2"),nt(D,ln)==0&&(e.a+=";!D0;!D1;!D2;!D3"),Rt(z(o,B1),0)&&h.Kkb(e,";D"+n.$o(t.k,A)),Rt(z(o,Z1),0)&&h.Kkb(e,";!D"+n.$o(t.k,A)),w.UAb(e.a).length==0?null:e.a;},n.qr=function(t){return t.p==null&&(t.p=n.kr(t)),t.p;},n.rr=function(t,A,e){var r,P,i,D,u;if(r=n._o(t.k,A,e),D=n.Zo(t.k,A,e),t.e[r]!=0){for(u=8*t.q[A]+1,P=s.xj(t.g,t.q[A]).c,i=0;i<P.length&&D!=P[i];i++)++u;return u;}return 8*t.q[D];},n.sr=function(t,A){var e,r,P,i,D;for(e=-1,r=-1,i=0;i<n.$o(t.k,A);i++)P=n.Zo(t.k,A,i),D=(n.ap(t.k,A,i)<<24)+t.a[P],!t.b[P]&&(e==-1||r<D)&&(e=i,r=D);return e;},n.tr=function(t,A){var e;for(e=0;e<n.$o(t.k,A);e++)if(!t.b[n.Zo(t.k,A,e)])return  true;return  false;},n.ur=function(t,A,e){var r,P;return r=s.xj(t.g,t.q[A]),r.g==e?false:(P=s.xj(t.g,t.q[e]),P.g==A?true:n.Zu(P,A));},n.vr=function(t,A){var e,r,P;for(r=false,e=1;e<4;e++)for(P=0;P<e;P++)t[P]>t[e]&&(r=!r),A[P]>A[e]&&(r=!r);return r;},n.wr=function(t){return t>=5&&t<=9||t>=15&&t<=17||t==35||t==53;},n.xr=function(t){n.yr.call(this,t,0);},n.yr=function(t,A){this.k=t,this.j=A;},n.zr=function(t){var A,e,r,P;for(P=new h.Mkb(),e=0;e<t.g.a.length;e++)e!=0&&(P.a+="."),h.Kkb(P,n.qr(new n.yr(s.xj(t.g,e),2)));for(P.a+=">",r=0;r<t.a.a.length;r++)r!=0&&(P.a+="."),h.Kkb(P,n.qr(new n.xr(s.xj(t.a,r))));for(P.a+=">",A=0;A<t.f.a.length;A++)A!=0&&(P.a+="."),h.Kkb(P,n.qr(new n.yr(s.xj(t.f,A),2)));return P.a;},I(111,1,{},n.xr,n.yr),L.j=0,Q.u8=N(111),n.Ar=function(t,A){this.b=t,this.a=A;},I(3,1,{3:1},n.Ar),L.a=0,L.b=0,Q.v8=N(3),n.Cr=function(){n.Cr=i1,n.Br=j(v(Q.v8,2),b0,17,0,[null,j(v(Q.v8,1),xt,3,0,[new n.Ar(0,1.007825032),new n.Ar(1,2.014101778),new n.Ar(2,3.016049268),new n.Ar(3,4.027834627),new n.Ar(4,5.039542911),new n.Ar(5,6.044942608)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(1,3.01602931),new n.Ar(2,4.00260325),new n.Ar(3,5.012223628),new n.Ar(4,6.018888072),new n.Ar(5,7.028030527),new n.Ar(6,8.033921838),new n.Ar(7,9.043820323),new n.Ar(8,10.052399713)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(1,4.027182329),new n.Ar(2,5.012537796),new n.Ar(3,6.015122281),new n.Ar(4,7.016004049),new n.Ar(5,8.02248667),new n.Ar(6,9.026789122),new n.Ar(7,10.035480884),new n.Ar(8,11.043796166),new n.Ar(9,12.05378)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(1,5.04079),new n.Ar(2,6.019725804),new n.Ar(3,7.016929246),new n.Ar(4,8.005305094),new n.Ar(5,9.012182135),new n.Ar(6,10.01353372),new n.Ar(7,11.021657653),new n.Ar(8,12.026920631),new n.Ar(9,13.036133834),new n.Ar(10,14.042815522)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(2,7.029917389),new n.Ar(3,8.024606713),new n.Ar(4,9.013328806),new n.Ar(5,10.012937027),new n.Ar(6,11.009305466),new n.Ar(7,12.014352109),new n.Ar(8,13.017780267),new n.Ar(9,14.025404064),new n.Ar(10,15.031097291),new n.Ar(11,16.039808836),new n.Ar(12,17.046931399),new n.Ar(13,18.05617),new n.Ar(14,19.06373)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(2,8.037675026),new n.Ar(3,9.031040087),new n.Ar(4,10.01685311),new n.Ar(5,11.011433818),new n.Ar(6,12),new n.Ar(7,13.003354838),new n.Ar(8,14.003241988),new n.Ar(9,15.010599258),new n.Ar(10,16.014701243),new n.Ar(11,17.022583712),new n.Ar(12,18.026757058),new n.Ar(13,19.035248094),new n.Ar(14,20.040322395),new n.Ar(15,21.04934),new n.Ar(16,22.05645)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(3,10.042618),new n.Ar(4,11.026796226),new n.Ar(5,12.018613202),new n.Ar(6,13.005738584),new n.Ar(7,14.003074005),new n.Ar(8,15.000108898),new n.Ar(9,16.006101417),new n.Ar(10,17.008449673),new n.Ar(11,18.014081827),new n.Ar(12,19.017026896),new n.Ar(13,20.023367295),new n.Ar(14,21.027087574),new n.Ar(15,22.034440259),new n.Ar(16,23.04051),new n.Ar(17,24.0505)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(4,12.034404776),new n.Ar(5,13.0248104),new n.Ar(6,14.008595285),new n.Ar(7,15.003065386),new n.Ar(8,15.994914622),new n.Ar(9,16.999131501),new n.Ar(10,17.999160419),new n.Ar(11,19.00357873),new n.Ar(12,20.00407615),new n.Ar(13,21.008654631),new n.Ar(14,22.009967157),new n.Ar(15,23.015691325),new n.Ar(16,24.020369922),new n.Ar(17,25.02914),new n.Ar(18,26.03775)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(5,14.03608),new n.Ar(6,15.018010856),new n.Ar(7,16.01146573),new n.Ar(8,17.002095238),new n.Ar(9,18.000937667),new n.Ar(10,18.998403205),new n.Ar(11,19.999981324),new n.Ar(12,20.999948921),new n.Ar(13,22.00299925),new n.Ar(14,23.003574385),new n.Ar(15,24.008099371),new n.Ar(16,25.012094963),new n.Ar(17,26.019633157),new n.Ar(18,27.026892316),new n.Ar(19,28.03567),new n.Ar(20,29.04326)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(6,16.025756907),new n.Ar(7,17.017697565),new n.Ar(8,18.005697066),new n.Ar(9,19.001879839),new n.Ar(10,19.992440176),new n.Ar(11,20.993846744),new n.Ar(12,21.99138551),new n.Ar(13,22.994467337),new n.Ar(14,23.993615074),new n.Ar(15,24.997789899),new n.Ar(16,26.000461498),new n.Ar(17,27.0076152),new n.Ar(18,28.012108072),new n.Ar(19,29.019345902),new n.Ar(20,30.023872),new n.Ar(21,31.03311),new n.Ar(22,32.03991)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(7,18.02718),new n.Ar(8,19.01387945),new n.Ar(9,20.00734826),new n.Ar(10,20.997655099),new n.Ar(11,21.994436782),new n.Ar(12,22.989769675),new n.Ar(13,23.990963332),new n.Ar(14,24.989954352),new n.Ar(15,25.992589898),new n.Ar(16,26.994008702),new n.Ar(17,27.99889041),new n.Ar(18,29.002811301),new n.Ar(19,30.009226487),new n.Ar(20,31.013595108),new n.Ar(21,32.019649792),new n.Ar(22,33.027386),new n.Ar(23,34.0349),new n.Ar(24,35.04418)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(8,20.018862744),new n.Ar(9,21.011714174),new n.Ar(10,21.999574055),new n.Ar(11,22.99412485),new n.Ar(12,23.985041898),new n.Ar(13,24.985837023),new n.Ar(14,25.98259304),new n.Ar(15,26.984340742),new n.Ar(16,27.983876703),new n.Ar(17,28.988554743),new n.Ar(18,29.990464529),new n.Ar(19,30.996548459),new n.Ar(20,31.999145889),new n.Ar(21,33.005586975),new n.Ar(22,34.00907244),new n.Ar(23,35.018669),new n.Ar(24,36.02245),new n.Ar(25,37.03124)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(8,21.02804),new n.Ar(9,22.01952),new n.Ar(10,23.0072649),new n.Ar(11,23.999940911),new n.Ar(12,24.990428555),new n.Ar(13,25.986891659),new n.Ar(14,26.981538441),new n.Ar(15,27.981910184),new n.Ar(16,28.980444848),new n.Ar(17,29.982960304),new n.Ar(18,30.983946023),new n.Ar(19,31.988124379),new n.Ar(20,32.990869587),new n.Ar(21,33.996927255),new n.Ar(22,34.99993765),new n.Ar(23,36.006351501),new n.Ar(24,37.01031),new n.Ar(25,38.0169),new n.Ar(26,39.0219)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(8,22.03453),new n.Ar(9,23.02552),new n.Ar(10,24.011545711),new n.Ar(11,25.00410664),new n.Ar(12,25.992329935),new n.Ar(13,26.986704764),new n.Ar(14,27.976926533),new n.Ar(15,28.976494719),new n.Ar(16,29.973770218),new n.Ar(17,30.975363275),new n.Ar(18,31.974148129),new n.Ar(19,32.97800052),new n.Ar(20,33.978575745),new n.Ar(21,34.984584158),new n.Ar(22,35.986687363),new n.Ar(23,36.99299599),new n.Ar(24,37.99598),new n.Ar(25,39.0023),new n.Ar(26,40.0058),new n.Ar(27,41.0127),new n.Ar(28,42.0161)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(9,24.03435),new n.Ar(10,25.02026),new n.Ar(11,26.01178),new n.Ar(12,26.999191645),new n.Ar(13,27.99231233),new n.Ar(14,28.981801376),new n.Ar(15,29.978313807),new n.Ar(16,30.973761512),new n.Ar(17,31.973907163),new n.Ar(18,32.971725281),new n.Ar(19,33.973636381),new n.Ar(20,34.973314249),new n.Ar(21,35.978259824),new n.Ar(22,36.979608338),new n.Ar(23,37.98447),new n.Ar(24,38.98642),new n.Ar(25,39.99105),new n.Ar(26,40.9948),new n.Ar(27,42.00009),new n.Ar(28,43.00331),new n.Ar(29,44.00988),new n.Ar(30,45.01514),new n.Ar(31,46.02383)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(10,26.02788),new n.Ar(11,27.018795),new n.Ar(12,28.004372661),new n.Ar(13,28.996608805),new n.Ar(14,29.984902954),new n.Ar(15,30.979554421),new n.Ar(16,31.97207069),new n.Ar(17,32.971458497),new n.Ar(18,33.967866831),new n.Ar(19,34.96903214),new n.Ar(20,35.96708088),new n.Ar(21,36.971125716),new n.Ar(22,37.971163443),new n.Ar(23,38.975135275),new n.Ar(24,39.97547),new n.Ar(25,40.98003),new n.Ar(26,41.98149),new n.Ar(27,42.9866),new n.Ar(28,43.98832),new n.Ar(29,44.99482),new n.Ar(30,45.99957),new n.Ar(31,47.00762),new n.Ar(32,48.01299),new n.Ar(33,49.02201)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(11,28.02851),new n.Ar(12,29.01411),new n.Ar(13,30.00477),new n.Ar(14,30.992416014),new n.Ar(15,31.985688908),new n.Ar(16,32.977451798),new n.Ar(17,33.973761967),new n.Ar(18,34.968852707),new n.Ar(19,35.968306945),new n.Ar(20,36.9659026),new n.Ar(21,37.96801055),new n.Ar(22,38.968007677),new n.Ar(23,39.970415555),new n.Ar(24,40.970650212),new n.Ar(25,41.973174994),new n.Ar(26,42.974203385),new n.Ar(27,43.978538712),new n.Ar(28,44.9797),new n.Ar(29,45.98412),new n.Ar(30,46.98795),new n.Ar(31,47.99485),new n.Ar(32,48.99989),new n.Ar(33,50.00773),new n.Ar(34,51.01353)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(12,30.02156),new n.Ar(13,31.012126),new n.Ar(14,31.99766066),new n.Ar(15,32.989928719),new n.Ar(16,33.980270118),new n.Ar(17,34.975256726),new n.Ar(18,35.967546282),new n.Ar(19,36.966775912),new n.Ar(20,37.962732161),new n.Ar(21,38.964313413),new n.Ar(22,39.962383123),new n.Ar(23,40.964500828),new n.Ar(24,41.963046386),new n.Ar(25,42.965670701),new n.Ar(26,43.965365269),new n.Ar(27,44.968094979),new n.Ar(28,45.968093467),new n.Ar(29,46.972186238),new n.Ar(30,47.97507),new n.Ar(31,48.98218),new n.Ar(32,49.98594),new n.Ar(33,50.99324),new n.Ar(34,51.99817),new n.Ar(35,53.006227)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(13,32.02192),new n.Ar(14,33.00726),new n.Ar(15,33.99841),new n.Ar(16,34.988011615),new n.Ar(17,35.981293405),new n.Ar(18,36.973376915),new n.Ar(19,37.969080107),new n.Ar(20,38.963706861),new n.Ar(21,39.963998672),new n.Ar(22,40.961825972),new n.Ar(23,41.962403059),new n.Ar(24,42.960715746),new n.Ar(25,43.961556146),new n.Ar(26,44.960699658),new n.Ar(27,45.961976203),new n.Ar(28,46.961677807),new n.Ar(29,47.965512946),new n.Ar(30,48.967450084),new n.Ar(31,49.972782832),new n.Ar(32,50.97638),new n.Ar(33,51.98261),new n.Ar(34,52.98712),new n.Ar(35,53.99399),new n.Ar(36,54.999388)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(14,34.01412),new n.Ar(15,35.004765),new n.Ar(16,35.993087234),new n.Ar(17,36.985871505),new n.Ar(18,37.976318637),new n.Ar(19,38.970717729),new n.Ar(20,39.962591155),new n.Ar(21,40.962278349),new n.Ar(22,41.958618337),new n.Ar(23,42.958766833),new n.Ar(24,43.955481094),new n.Ar(25,44.956185938),new n.Ar(26,45.953692759),new n.Ar(27,46.954546459),new n.Ar(28,47.952533512),new n.Ar(29,48.955673302),new n.Ar(30,49.957518286),new n.Ar(31,50.961474238),new n.Ar(32,51.9651),new n.Ar(33,52.97005),new n.Ar(34,53.97468),new n.Ar(35,54.98055),new n.Ar(36,55.98579),new n.Ar(37,56.992356)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(15,36.01492),new n.Ar(16,37.00305),new n.Ar(17,37.9947),new n.Ar(18,38.984790009),new n.Ar(19,39.977964014),new n.Ar(20,40.969251316),new n.Ar(21,41.965516761),new n.Ar(22,42.96115098),new n.Ar(23,43.959403048),new n.Ar(24,44.955910243),new n.Ar(25,45.95517025),new n.Ar(26,46.952408027),new n.Ar(27,47.952234991),new n.Ar(28,48.950024065),new n.Ar(29,49.952187008),new n.Ar(30,50.9536027),new n.Ar(31,51.95665),new n.Ar(32,52.95817),new n.Ar(33,53.963),new n.Ar(34,54.9694),new n.Ar(35,55.97266),new n.Ar(36,56.97704),new n.Ar(37,57.98307),new n.Ar(38,58.988041)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(16,38.00977),new n.Ar(17,39.001323),new n.Ar(18,39.990498907),new n.Ar(19,40.983131),new n.Ar(20,41.973031622),new n.Ar(21,42.968523342),new n.Ar(22,43.959690235),new n.Ar(23,44.958124349),new n.Ar(24,45.952629491),new n.Ar(25,46.951763792),new n.Ar(26,47.947947053),new n.Ar(27,48.947870789),new n.Ar(28,49.944792069),new n.Ar(29,50.946616017),new n.Ar(30,51.946898175),new n.Ar(31,52.949731709),new n.Ar(32,53.95087),new n.Ar(33,54.95512),new n.Ar(34,55.95799),new n.Ar(35,56.9643),new n.Ar(36,57.96611),new n.Ar(37,58.97196),new n.Ar(38,59.97564),new n.Ar(39,60.982018)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(17,40.01109),new n.Ar(18,40.99974),new n.Ar(19,41.99123),new n.Ar(20,42.98065),new n.Ar(21,43.9744),new n.Ar(22,44.965782286),new n.Ar(23,45.960199491),new n.Ar(24,46.954906918),new n.Ar(25,47.95225448),new n.Ar(26,48.948516914),new n.Ar(27,49.947162792),new n.Ar(28,50.943963675),new n.Ar(29,51.944779658),new n.Ar(30,52.944342517),new n.Ar(31,53.946444381),new n.Ar(32,54.947238194),new n.Ar(33,55.95036),new n.Ar(34,56.95236),new n.Ar(35,57.95665),new n.Ar(36,58.9593),new n.Ar(37,59.9645),new n.Ar(38,60.96741),new n.Ar(39,61.97314),new n.Ar(40,62.97675)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(18,42.00643),new n.Ar(19,42.997707),new n.Ar(20,43.98547),new n.Ar(21,44.97916),new n.Ar(22,45.968361649),new n.Ar(23,46.962906512),new n.Ar(24,47.954035861),new n.Ar(25,48.951341135),new n.Ar(26,49.946049607),new n.Ar(27,50.944771767),new n.Ar(28,51.940511904),new n.Ar(29,52.940653781),new n.Ar(30,53.938884921),new n.Ar(31,54.940844164),new n.Ar(32,55.940645238),new n.Ar(33,56.9437538),new n.Ar(34,57.94425),new n.Ar(35,58.94863),new n.Ar(36,59.94973),new n.Ar(37,60.95409),new n.Ar(38,61.9558),new n.Ar(39,62.96186),new n.Ar(40,63.9642),new n.Ar(41,64.97037)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(19,44.00687),new n.Ar(20,44.99451),new n.Ar(21,45.98672),new n.Ar(22,46.9761),new n.Ar(23,47.96887),new n.Ar(24,48.959623415),new n.Ar(25,49.95424396),new n.Ar(26,50.948215487),new n.Ar(27,51.945570079),new n.Ar(28,52.941294702),new n.Ar(29,53.940363247),new n.Ar(30,54.938049636),new n.Ar(31,55.938909366),new n.Ar(32,56.938287458),new n.Ar(33,57.939986451),new n.Ar(34,58.940447166),new n.Ar(35,59.943193998),new n.Ar(36,60.94446),new n.Ar(37,61.94797),new n.Ar(38,62.94981),new n.Ar(39,63.95373),new n.Ar(40,64.9561),new n.Ar(41,65.96082),new n.Ar(42,66.96382)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(19,45.01456),new n.Ar(20,46.00081),new n.Ar(21,46.99289),new n.Ar(22,47.98056),new n.Ar(23,48.97361),new n.Ar(24,49.962993316),new n.Ar(25,50.956824936),new n.Ar(26,51.948116526),new n.Ar(27,52.945312282),new n.Ar(28,53.939614836),new n.Ar(29,54.938298029),new n.Ar(30,55.934942133),new n.Ar(31,56.935398707),new n.Ar(32,57.933280458),new n.Ar(33,58.934880493),new n.Ar(34,59.934076943),new n.Ar(35,60.936749461),new n.Ar(36,61.936770495),new n.Ar(37,62.940118442),new n.Ar(38,63.94087),new n.Ar(39,64.94494),new n.Ar(40,65.94598),new n.Ar(41,66.95),new n.Ar(42,67.95251),new n.Ar(43,68.9577)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(21,48.00176),new n.Ar(22,48.98972),new n.Ar(23,49.98154),new n.Ar(24,50.97072),new n.Ar(25,51.96359),new n.Ar(26,52.954224985),new n.Ar(27,53.948464147),new n.Ar(28,54.942003149),new n.Ar(29,55.939843937),new n.Ar(30,56.936296235),new n.Ar(31,57.935757571),new n.Ar(32,58.933200194),new n.Ar(33,59.933822196),new n.Ar(34,60.932479381),new n.Ar(35,61.934054212),new n.Ar(36,62.933615218),new n.Ar(37,63.935813523),new n.Ar(38,64.936484581),new n.Ar(39,65.939825412),new n.Ar(40,66.94061),new n.Ar(41,67.94436),new n.Ar(42,68.9452),new n.Ar(43,69.94981),new n.Ar(44,70.95173),new n.Ar(45,71.95641)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(22,49.99593),new n.Ar(23,50.98772),new n.Ar(24,51.97568),new n.Ar(25,52.96846),new n.Ar(26,53.957910508),new n.Ar(27,54.951336329),new n.Ar(28,55.942136339),new n.Ar(29,56.939800489),new n.Ar(30,57.935347922),new n.Ar(31,58.934351553),new n.Ar(32,59.930790633),new n.Ar(33,60.931060442),new n.Ar(34,61.928348763),new n.Ar(35,62.929672948),new n.Ar(36,63.927969574),new n.Ar(37,64.930088013),new n.Ar(38,65.929115232),new n.Ar(39,66.931569638),new n.Ar(40,67.931844932),new n.Ar(41,68.935181837),new n.Ar(42,69.93614),new n.Ar(43,70.94),new n.Ar(44,71.9413),new n.Ar(45,72.94608),new n.Ar(46,73.94791),new n.Ar(47,74.95297),new n.Ar(48,75.95533),new n.Ar(49,76.96083),new n.Ar(50,77.9638)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(23,51.99718),new n.Ar(24,52.98555),new n.Ar(25,53.97671),new n.Ar(26,54.96605),new n.Ar(27,55.95856),new n.Ar(28,56.949215695),new n.Ar(29,57.944540734),new n.Ar(30,58.939504114),new n.Ar(31,59.937368123),new n.Ar(32,60.933462181),new n.Ar(33,61.932587299),new n.Ar(34,62.929601079),new n.Ar(35,63.929767865),new n.Ar(36,64.927793707),new n.Ar(37,65.928873041),new n.Ar(38,66.927750294),new n.Ar(39,67.929637875),new n.Ar(40,68.929425281),new n.Ar(41,69.932409287),new n.Ar(42,70.932619818),new n.Ar(43,71.93552),new n.Ar(44,72.93649),new n.Ar(45,73.9402),new n.Ar(46,74.9417),new n.Ar(47,75.94599),new n.Ar(48,76.94795),new n.Ar(49,77.95281),new n.Ar(50,78.95528),new n.Ar(51,79.96189)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(24,53.99295),new n.Ar(25,54.98398),new n.Ar(26,55.97238),new n.Ar(27,56.96491),new n.Ar(28,57.954596465),new n.Ar(29,58.949267074),new n.Ar(30,59.941832031),new n.Ar(31,60.939513907),new n.Ar(32,61.934334132),new n.Ar(33,62.933215563),new n.Ar(34,63.929146578),new n.Ar(35,64.929245079),new n.Ar(36,65.926036763),new n.Ar(37,66.927130859),new n.Ar(38,67.924847566),new n.Ar(39,68.926553538),new n.Ar(40,69.92532487),new n.Ar(41,70.927727195),new n.Ar(42,71.926861122),new n.Ar(43,72.929779469),new n.Ar(44,73.929458261),new n.Ar(45,74.932937379),new n.Ar(46,75.933394207),new n.Ar(47,76.937085857),new n.Ar(48,77.938569576),new n.Ar(49,78.942095175),new n.Ar(50,79.944414722),new n.Ar(51,80.95048),new n.Ar(52,81.95484)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(25,55.99491),new n.Ar(26,56.98293),new n.Ar(27,57.97425),new n.Ar(28,58.96337),new n.Ar(29,59.95706),new n.Ar(30,60.94917),new n.Ar(31,61.944179608),new n.Ar(32,62.939141527),new n.Ar(33,63.936838307),new n.Ar(34,64.932739322),new n.Ar(35,65.931592355),new n.Ar(36,66.928204915),new n.Ar(37,67.927983497),new n.Ar(38,68.925580912),new n.Ar(39,69.926027741),new n.Ar(40,70.92470501),new n.Ar(41,71.92636935),new n.Ar(42,72.925169832),new n.Ar(43,73.926940999),new n.Ar(44,74.926500645),new n.Ar(45,75.928928262),new n.Ar(46,76.929281189),new n.Ar(47,77.93165595),new n.Ar(48,78.932916371),new n.Ar(49,79.936588154),new n.Ar(50,80.937752955),new n.Ar(51,81.94316),new n.Ar(52,82.94687),new n.Ar(53,83.95234)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(26,57.99101),new n.Ar(27,58.98175),new n.Ar(28,59.97019),new n.Ar(29,60.96379),new n.Ar(30,61.95465),new n.Ar(31,62.94964),new n.Ar(32,63.941572638),new n.Ar(33,64.939440762),new n.Ar(34,65.933846798),new n.Ar(35,66.932738415),new n.Ar(36,67.928097266),new n.Ar(37,68.927972002),new n.Ar(38,69.924250365),new n.Ar(39,70.924953991),new n.Ar(40,71.922076184),new n.Ar(41,72.923459361),new n.Ar(42,73.921178213),new n.Ar(43,74.922859494),new n.Ar(44,75.921402716),new n.Ar(45,76.923548462),new n.Ar(46,77.922852886),new n.Ar(47,78.92540156),new n.Ar(48,79.925444764),new n.Ar(49,80.928821065),new n.Ar(50,81.929550326),new n.Ar(51,82.93451),new n.Ar(52,83.93731),new n.Ar(53,84.94269),new n.Ar(54,85.94627)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(27,59.99313),new n.Ar(28,60.98062),new n.Ar(29,61.9732),new n.Ar(30,62.96369),new n.Ar(31,63.957572),new n.Ar(32,64.949484),new n.Ar(33,65.944099147),new n.Ar(34,66.939190417),new n.Ar(35,67.936792976),new n.Ar(36,68.932280154),new n.Ar(37,69.930927811),new n.Ar(38,70.927114724),new n.Ar(39,71.926752647),new n.Ar(40,72.923825288),new n.Ar(41,73.923929076),new n.Ar(42,74.921596417),new n.Ar(43,75.922393933),new n.Ar(44,76.920647703),new n.Ar(45,77.921828577),new n.Ar(46,78.920948498),new n.Ar(47,79.922578162),new n.Ar(48,80.922132884),new n.Ar(49,81.924504668),new n.Ar(50,82.924980625),new n.Ar(51,83.92906),new n.Ar(52,84.93181),new n.Ar(53,85.93623),new n.Ar(54,86.93958),new n.Ar(55,87.94456),new n.Ar(56,88.94923)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(31,64.96466),new n.Ar(32,65.95521),new n.Ar(33,66.95009),new n.Ar(34,67.94187),new n.Ar(35,68.939562155),new n.Ar(36,69.933504),new n.Ar(37,70.931868378),new n.Ar(38,71.927112313),new n.Ar(39,72.9267668),new n.Ar(40,73.922476561),new n.Ar(41,74.922523571),new n.Ar(42,75.919214107),new n.Ar(43,76.91991461),new n.Ar(44,77.917309522),new n.Ar(45,78.918499802),new n.Ar(46,79.916521828),new n.Ar(47,80.917992931),new n.Ar(48,81.9167),new n.Ar(49,82.919119072),new n.Ar(50,83.918464523),new n.Ar(51,84.922244678),new n.Ar(52,85.924271165),new n.Ar(53,86.928520749),new n.Ar(54,87.931423982),new n.Ar(55,88.93602),new n.Ar(56,89.93942),new n.Ar(57,90.94537),new n.Ar(58,91.94933)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(32,66.96479),new n.Ar(33,67.958248),new n.Ar(34,68.950178),new n.Ar(35,69.944208),new n.Ar(36,70.939246),new n.Ar(37,71.936496876),new n.Ar(38,72.931794889),new n.Ar(39,73.929891152),new n.Ar(40,74.92577641),new n.Ar(41,75.924541974),new n.Ar(42,76.921380123),new n.Ar(43,77.92114613),new n.Ar(44,78.918337647),new n.Ar(45,79.918529952),new n.Ar(46,80.91629106),new n.Ar(47,81.916804666),new n.Ar(48,82.915180219),new n.Ar(49,83.916503685),new n.Ar(50,84.915608027),new n.Ar(51,85.918797162),new n.Ar(52,86.920710713),new n.Ar(53,87.924065908),new n.Ar(54,88.92638726),new n.Ar(55,89.930634988),new n.Ar(56,90.9339653),new n.Ar(57,91.939255258),new n.Ar(58,92.9431),new n.Ar(59,93.94868)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(33,68.96532),new n.Ar(34,69.95601),new n.Ar(35,70.95051),new n.Ar(36,71.94190754),new n.Ar(37,72.938931115),new n.Ar(38,73.933258225),new n.Ar(39,74.931033794),new n.Ar(40,75.925948304),new n.Ar(41,76.92466788),new n.Ar(42,77.920386271),new n.Ar(43,78.920082992),new n.Ar(44,79.91637804),new n.Ar(45,80.916592419),new n.Ar(46,81.913484601),new n.Ar(47,82.914135952),new n.Ar(48,83.911506627),new n.Ar(49,84.912526954),new n.Ar(50,85.910610313),new n.Ar(51,86.913354251),new n.Ar(52,87.914446951),new n.Ar(53,88.917632505),new n.Ar(54,89.919523803),new n.Ar(55,90.923442418),new n.Ar(56,91.926152752),new n.Ar(57,92.931265246),new n.Ar(58,93.934362),new n.Ar(59,94.93984),new n.Ar(60,95.94307),new n.Ar(61,96.94856)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(34,70.96532),new n.Ar(35,71.95908),new n.Ar(36,72.950366),new n.Ar(37,73.944470376),new n.Ar(38,74.938569199),new n.Ar(39,75.935071448),new n.Ar(40,76.930406599),new n.Ar(41,77.928141485),new n.Ar(42,78.923996719),new n.Ar(43,79.922519322),new n.Ar(44,80.918994165),new n.Ar(45,81.918207691),new n.Ar(46,82.915111951),new n.Ar(47,83.914384676),new n.Ar(48,84.911789341),new n.Ar(49,85.91116708),new n.Ar(50,86.909183465),new n.Ar(51,87.911318556),new n.Ar(52,88.912279939),new n.Ar(53,89.914808941),new n.Ar(54,90.91653416),new n.Ar(55,91.919725442),new n.Ar(56,92.922032765),new n.Ar(57,93.926407326),new n.Ar(58,94.92931926),new n.Ar(59,95.934283962),new n.Ar(60,96.937342863),new n.Ar(61,97.941703557),new n.Ar(62,98.945420616),new n.Ar(63,99.94987),new n.Ar(64,100.953195994),new n.Ar(65,101.95921)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(35,72.96597),new n.Ar(36,73.95631),new n.Ar(37,74.94992),new n.Ar(38,75.94161),new n.Ar(39,76.937761511),new n.Ar(40,77.932179362),new n.Ar(41,78.929707076),new n.Ar(42,79.924524588),new n.Ar(43,80.923213095),new n.Ar(44,81.918401258),new n.Ar(45,82.917555029),new n.Ar(46,83.913424778),new n.Ar(47,84.912932689),new n.Ar(48,85.909262351),new n.Ar(49,86.908879316),new n.Ar(50,87.905614339),new n.Ar(51,88.907452906),new n.Ar(52,89.907737596),new n.Ar(53,90.910209845),new n.Ar(54,91.911029895),new n.Ar(55,92.91402241),new n.Ar(56,93.915359856),new n.Ar(57,94.919358213),new n.Ar(58,95.921680473),new n.Ar(59,96.926148757),new n.Ar(60,97.928471177),new n.Ar(61,98.933315038),new n.Ar(62,99.935351729),new n.Ar(63,100.940517434),new n.Ar(64,101.943018795),new n.Ar(65,102.94895),new n.Ar(66,103.95233)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(38,76.94962),new n.Ar(39,77.9435),new n.Ar(40,78.937350712),new n.Ar(41,79.931982402),new n.Ar(42,80.929128719),new n.Ar(43,81.926792071),new n.Ar(44,82.922352572),new n.Ar(45,83.920387768),new n.Ar(46,84.916427076),new n.Ar(47,85.914887724),new n.Ar(48,86.910877833),new n.Ar(49,87.909503361),new n.Ar(50,88.905847902),new n.Ar(51,89.907151443),new n.Ar(52,90.907303415),new n.Ar(53,91.908946832),new n.Ar(54,92.909581582),new n.Ar(55,93.911594008),new n.Ar(56,94.912823709),new n.Ar(57,95.915897787),new n.Ar(58,96.918131017),new n.Ar(59,97.922219525),new n.Ar(60,98.924634736),new n.Ar(61,99.927756402),new n.Ar(62,100.930313395),new n.Ar(63,101.933555501),new n.Ar(64,102.93694),new n.Ar(65,103.94145),new n.Ar(66,104.94509),new n.Ar(67,105.95022)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(39,78.94916),new n.Ar(40,79.94055),new n.Ar(41,80.936815296),new n.Ar(42,81.931086249),new n.Ar(43,82.92865213),new n.Ar(44,83.92325),new n.Ar(45,84.92146522),new n.Ar(46,85.916472851),new n.Ar(47,86.914816578),new n.Ar(48,87.910226179),new n.Ar(49,88.908888916),new n.Ar(50,89.904703679),new n.Ar(51,90.905644968),new n.Ar(52,91.905040106),new n.Ar(53,92.906475627),new n.Ar(54,93.906315765),new n.Ar(55,94.908042739),new n.Ar(56,95.908275675),new n.Ar(57,96.910950716),new n.Ar(58,97.912746366),new n.Ar(59,98.916511084),new n.Ar(60,99.917761704),new n.Ar(61,100.921139958),new n.Ar(62,101.922981089),new n.Ar(63,102.926597062),new n.Ar(64,103.92878),new n.Ar(65,104.93305),new n.Ar(66,105.93591),new n.Ar(67,106.94086),new n.Ar(68,107.94428)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(40,80.94905),new n.Ar(41,81.94313),new n.Ar(42,82.936703713),new n.Ar(43,83.93357),new n.Ar(44,84.927906486),new n.Ar(45,85.925037588),new n.Ar(46,86.920361435),new n.Ar(47,87.91833144),new n.Ar(48,88.913495503),new n.Ar(49,89.911264109),new n.Ar(50,90.906990538),new n.Ar(51,91.907193214),new n.Ar(52,92.906377543),new n.Ar(53,93.907283457),new n.Ar(54,94.906835178),new n.Ar(55,95.908100076),new n.Ar(56,96.908097144),new n.Ar(57,97.91033069),new n.Ar(58,98.911617864),new n.Ar(59,99.914181434),new n.Ar(60,100.915251567),new n.Ar(61,101.918037417),new n.Ar(62,102.919141297),new n.Ar(63,103.922459464),new n.Ar(64,104.923934023),new n.Ar(65,105.92819),new n.Ar(66,106.93031),new n.Ar(67,107.93501),new n.Ar(68,108.93763),new n.Ar(69,109.94268)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(41,82.94874),new n.Ar(42,83.94009),new n.Ar(43,84.93659),new n.Ar(44,85.930695167),new n.Ar(45,86.92732683),new n.Ar(46,87.921952728),new n.Ar(47,88.919480562),new n.Ar(48,89.913936161),new n.Ar(49,90.911750754),new n.Ar(50,91.90681048),new n.Ar(51,92.906812213),new n.Ar(52,93.905087578),new n.Ar(53,94.905841487),new n.Ar(54,95.904678904),new n.Ar(55,96.906021033),new n.Ar(56,97.905407846),new n.Ar(57,98.907711598),new n.Ar(58,99.907477149),new n.Ar(59,100.910346543),new n.Ar(60,101.910297162),new n.Ar(61,102.913204596),new n.Ar(62,103.913758387),new n.Ar(63,104.916972087),new n.Ar(64,105.918134284),new n.Ar(65,106.921694724),new n.Ar(66,107.923973837),new n.Ar(67,108.92781),new n.Ar(68,109.92973),new n.Ar(69,110.93451),new n.Ar(70,111.93684),new n.Ar(71,112.94203)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(42,84.94894),new n.Ar(43,85.94288),new n.Ar(44,86.93653),new n.Ar(45,87.93283),new n.Ar(46,88.92754288),new n.Ar(47,89.92355583),new n.Ar(48,90.9184282),new n.Ar(49,91.915259655),new n.Ar(50,92.910248473),new n.Ar(51,93.909656309),new n.Ar(52,94.907656454),new n.Ar(53,95.907870803),new n.Ar(54,96.906364843),new n.Ar(55,97.907215692),new n.Ar(56,98.906254554),new n.Ar(57,99.907657594),new n.Ar(58,100.90731438),new n.Ar(59,101.909212938),new n.Ar(60,102.909178805),new n.Ar(61,103.911444898),new n.Ar(62,104.911658043),new n.Ar(63,105.914355408),new n.Ar(64,106.915081691),new n.Ar(65,107.918479973),new n.Ar(66,108.919980998),new n.Ar(67,109.92339),new n.Ar(68,110.92505),new n.Ar(69,111.92924),new n.Ar(70,112.93133),new n.Ar(71,113.93588),new n.Ar(72,114.93828)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(43,86.94918),new n.Ar(44,87.94042),new n.Ar(45,88.93611),new n.Ar(46,89.92978),new n.Ar(47,90.926377434),new n.Ar(48,91.92012),new n.Ar(49,92.917051523),new n.Ar(50,93.911359569),new n.Ar(51,94.910412729),new n.Ar(52,95.907597681),new n.Ar(53,96.907554546),new n.Ar(54,97.905287111),new n.Ar(55,98.905939307),new n.Ar(56,99.904219664),new n.Ar(57,100.905582219),new n.Ar(58,101.904349503),new n.Ar(59,102.906323677),new n.Ar(60,103.905430145),new n.Ar(61,104.907750341),new n.Ar(62,105.907326913),new n.Ar(63,106.909907207),new n.Ar(64,107.910192211),new n.Ar(65,108.913201565),new n.Ar(66,109.913966185),new n.Ar(67,110.91756),new n.Ar(68,111.918821673),new n.Ar(69,112.92254),new n.Ar(70,113.923891981),new n.Ar(71,114.92831),new n.Ar(72,115.93016),new n.Ar(73,116.93479),new n.Ar(74,117.93703)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(44,88.94938),new n.Ar(45,89.94287),new n.Ar(46,90.93655),new n.Ar(47,91.93198),new n.Ar(48,92.92574),new n.Ar(49,93.921698),new n.Ar(50,94.915898541),new n.Ar(51,95.914518212),new n.Ar(52,96.911336643),new n.Ar(53,97.910716431),new n.Ar(54,98.908132101),new n.Ar(55,99.90811663),new n.Ar(56,100.906163526),new n.Ar(57,101.906842845),new n.Ar(58,102.905504182),new n.Ar(59,103.906655315),new n.Ar(60,104.905692444),new n.Ar(61,105.907284615),new n.Ar(62,106.90675054),new n.Ar(63,107.908730768),new n.Ar(64,108.908735621),new n.Ar(65,109.910949525),new n.Ar(66,110.91166),new n.Ar(67,111.913969253),new n.Ar(68,112.91542),new n.Ar(69,113.91734336),new n.Ar(70,114.920124676),new n.Ar(71,115.922746643),new n.Ar(72,116.92535),new n.Ar(73,117.92943),new n.Ar(74,118.93136),new n.Ar(75,119.93578),new n.Ar(76,120.93808)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(45,90.94948),new n.Ar(46,91.94042),new n.Ar(47,92.93591),new n.Ar(48,93.92877),new n.Ar(49,94.92469),new n.Ar(50,95.91822194),new n.Ar(51,96.916478921),new n.Ar(52,97.912720751),new n.Ar(53,98.911767757),new n.Ar(54,99.908504596),new n.Ar(55,100.908289144),new n.Ar(56,101.905607716),new n.Ar(57,102.906087204),new n.Ar(58,103.904034912),new n.Ar(59,104.905084046),new n.Ar(60,105.903483087),new n.Ar(61,106.905128453),new n.Ar(62,107.903894451),new n.Ar(63,108.905953535),new n.Ar(64,109.905152385),new n.Ar(65,110.907643952),new n.Ar(66,111.907313277),new n.Ar(67,112.910151346),new n.Ar(68,113.910365322),new n.Ar(69,114.91368341),new n.Ar(70,115.914158288),new n.Ar(71,116.91784),new n.Ar(72,117.918983915),new n.Ar(73,118.92268),new n.Ar(74,119.92403),new n.Ar(75,120.92818),new n.Ar(76,121.9298),new n.Ar(77,122.93426)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(47,93.94278),new n.Ar(48,94.93548),new n.Ar(49,95.93068),new n.Ar(50,96.924),new n.Ar(51,97.921759995),new n.Ar(52,98.917597103),new n.Ar(53,99.916069387),new n.Ar(54,100.912802135),new n.Ar(55,101.911999996),new n.Ar(56,102.908972453),new n.Ar(57,103.908628228),new n.Ar(58,104.906528234),new n.Ar(59,105.906666431),new n.Ar(60,106.90509302),new n.Ar(61,107.905953705),new n.Ar(62,108.904755514),new n.Ar(63,109.90611046),new n.Ar(64,110.905294679),new n.Ar(65,111.907004132),new n.Ar(66,112.906565708),new n.Ar(67,113.908807907),new n.Ar(68,114.908762282),new n.Ar(69,115.911359558),new n.Ar(70,116.911684187),new n.Ar(71,117.914582383),new n.Ar(72,118.915666045),new n.Ar(73,119.918788609),new n.Ar(74,120.919851074),new n.Ar(75,121.92332),new n.Ar(76,122.9249),new n.Ar(77,123.92853),new n.Ar(78,124.93054),new n.Ar(79,125.9345),new n.Ar(80,126.93688)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(48,95.93977),new n.Ar(49,96.93494),new n.Ar(50,97.927579),new n.Ar(51,98.92501),new n.Ar(52,99.920230232),new n.Ar(53,100.918681442),new n.Ar(54,101.914777255),new n.Ar(55,102.913418952),new n.Ar(56,103.909848091),new n.Ar(57,104.909467818),new n.Ar(58,105.906458007),new n.Ar(59,106.906614232),new n.Ar(60,107.904183403),new n.Ar(61,108.904985569),new n.Ar(62,109.903005578),new n.Ar(63,110.904181628),new n.Ar(64,111.902757226),new n.Ar(65,112.904400947),new n.Ar(66,113.903358121),new n.Ar(67,114.905430553),new n.Ar(68,115.904755434),new n.Ar(69,116.907218242),new n.Ar(70,117.906914144),new n.Ar(71,118.909922582),new n.Ar(72,119.909851352),new n.Ar(73,120.91298039),new n.Ar(74,121.9135),new n.Ar(75,122.917003675),new n.Ar(76,123.917648302),new n.Ar(77,124.92124717),new n.Ar(78,125.922353996),new n.Ar(79,126.926434822),new n.Ar(80,127.927760617),new n.Ar(81,128.93226),new n.Ar(82,129.93398)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(49,97.94224),new n.Ar(50,98.93461),new n.Ar(51,99.931149033),new n.Ar(52,100.92656),new n.Ar(53,101.924707541),new n.Ar(54,102.919913896),new n.Ar(55,103.918338416),new n.Ar(56,104.914673434),new n.Ar(57,105.913461134),new n.Ar(58,106.910292195),new n.Ar(59,107.909719683),new n.Ar(60,108.907154078),new n.Ar(61,109.907168783),new n.Ar(62,110.905110677),new n.Ar(63,111.905533338),new n.Ar(64,112.904061223),new n.Ar(65,113.904916758),new n.Ar(66,114.903878328),new n.Ar(67,115.905259995),new n.Ar(68,116.904515731),new n.Ar(69,117.906354623),new n.Ar(70,118.905846334),new n.Ar(71,119.907961505),new n.Ar(72,120.907848847),new n.Ar(73,121.910277103),new n.Ar(74,122.910438951),new n.Ar(75,123.913175916),new n.Ar(76,124.913601387),new n.Ar(77,125.916464532),new n.Ar(78,126.917344048),new n.Ar(79,127.920170658),new n.Ar(80,128.921657958),new n.Ar(81,129.924854941),new n.Ar(82,130.926767408),new n.Ar(83,131.932919005),new n.Ar(84,132.93834),new n.Ar(85,133.94466)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(50,99.938954),new n.Ar(51,100.93606),new n.Ar(52,101.93049),new n.Ar(53,102.92813),new n.Ar(54,103.923185469),new n.Ar(55,104.921390409),new n.Ar(56,105.916880472),new n.Ar(57,106.915666702),new n.Ar(58,107.911965339),new n.Ar(59,108.911286879),new n.Ar(60,109.907852688),new n.Ar(61,110.907735404),new n.Ar(62,111.90482081),new n.Ar(63,112.905173373),new n.Ar(64,113.902781816),new n.Ar(65,114.903345973),new n.Ar(66,115.901744149),new n.Ar(67,116.902953765),new n.Ar(68,117.901606328),new n.Ar(69,118.90330888),new n.Ar(70,119.902196571),new n.Ar(71,120.904236867),new n.Ar(72,121.903440138),new n.Ar(73,122.905721901),new n.Ar(74,123.90527463),new n.Ar(75,124.907784924),new n.Ar(76,125.907653953),new n.Ar(77,126.91035098),new n.Ar(78,127.910534953),new n.Ar(79,128.913439976),new n.Ar(80,129.913852185),new n.Ar(81,130.916919144),new n.Ar(82,131.917744455),new n.Ar(83,132.923814085),new n.Ar(84,133.928463576),new n.Ar(85,134.93473),new n.Ar(86,135.93934),new n.Ar(87,136.94579)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(52,102.94012),new n.Ar(53,103.936287),new n.Ar(54,104.931528593),new n.Ar(55,105.928183134),new n.Ar(56,106.92415),new n.Ar(57,107.92216),new n.Ar(58,108.918136092),new n.Ar(59,109.917533911),new n.Ar(60,110.912534147),new n.Ar(61,111.91239464),new n.Ar(62,112.909377941),new n.Ar(63,113.909095876),new n.Ar(64,114.906598812),new n.Ar(65,115.906797235),new n.Ar(66,116.90483959),new n.Ar(67,117.905531885),new n.Ar(68,118.90394646),new n.Ar(69,119.905074315),new n.Ar(70,120.903818044),new n.Ar(71,121.905175415),new n.Ar(72,122.904215696),new n.Ar(73,123.905937525),new n.Ar(74,124.905247804),new n.Ar(75,125.907248153),new n.Ar(76,126.906914564),new n.Ar(77,127.90916733),new n.Ar(78,128.909150092),new n.Ar(79,129.911546459),new n.Ar(80,130.911946487),new n.Ar(81,131.914413247),new n.Ar(82,132.915236466),new n.Ar(83,133.920551554),new n.Ar(84,134.925167962),new n.Ar(85,135.93066),new n.Ar(86,136.93531),new n.Ar(87,137.94096),new n.Ar(88,138.94571)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(54,105.937702),new n.Ar(55,106.935036),new n.Ar(56,107.929486838),new n.Ar(57,108.927456483),new n.Ar(58,109.922407164),new n.Ar(59,110.921120589),new n.Ar(60,111.917061617),new n.Ar(61,112.915452551),new n.Ar(62,113.912498025),new n.Ar(63,114.911578627),new n.Ar(64,115.908420253),new n.Ar(65,116.90863418),new n.Ar(66,117.905825187),new n.Ar(67,118.90640811),new n.Ar(68,119.904019891),new n.Ar(69,120.904929815),new n.Ar(70,121.903047064),new n.Ar(71,122.904272951),new n.Ar(72,123.902819466),new n.Ar(73,124.904424718),new n.Ar(74,125.903305543),new n.Ar(75,126.90521729),new n.Ar(76,127.904461383),new n.Ar(77,128.906595593),new n.Ar(78,129.906222753),new n.Ar(79,130.90852188),new n.Ar(80,131.908523782),new n.Ar(81,132.910939068),new n.Ar(82,133.911540546),new n.Ar(83,134.916450782),new n.Ar(84,135.920103155),new n.Ar(85,136.925324769),new n.Ar(86,137.92922),new n.Ar(87,138.93473),new n.Ar(88,139.9387),new n.Ar(89,140.94439),new n.Ar(90,141.9485)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(55,107.943291),new n.Ar(56,108.938191658),new n.Ar(57,109.934634181),new n.Ar(58,110.930276),new n.Ar(59,111.92797),new n.Ar(60,112.923644245),new n.Ar(61,113.92185),new n.Ar(62,114.918272),new n.Ar(63,115.916735014),new n.Ar(64,116.913647692),new n.Ar(65,117.91337523),new n.Ar(66,118.910180837),new n.Ar(67,119.910047843),new n.Ar(68,120.907366063),new n.Ar(69,121.907592451),new n.Ar(70,122.905597944),new n.Ar(71,123.906211423),new n.Ar(72,124.90462415),new n.Ar(73,125.905619387),new n.Ar(74,126.90446842),new n.Ar(75,127.905805254),new n.Ar(76,128.904987487),new n.Ar(77,129.906674018),new n.Ar(78,130.906124168),new n.Ar(79,131.907994525),new n.Ar(80,132.907806465),new n.Ar(81,133.909876552),new n.Ar(82,134.91005031),new n.Ar(83,135.914655105),new n.Ar(84,136.917872653),new n.Ar(85,137.922383666),new n.Ar(86,138.926093402),new n.Ar(87,139.93121),new n.Ar(88,140.93483),new n.Ar(89,141.94018),new n.Ar(90,142.94407),new n.Ar(91,143.94961)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(56,109.944476),new n.Ar(57,110.941632),new n.Ar(58,111.93566535),new n.Ar(59,112.933382836),new n.Ar(60,113.928145),new n.Ar(61,114.926979032),new n.Ar(62,115.921394197),new n.Ar(63,116.920564355),new n.Ar(64,117.91657092),new n.Ar(65,118.915554295),new n.Ar(66,119.91215199),new n.Ar(67,120.911386497),new n.Ar(68,121.908548396),new n.Ar(69,122.908470748),new n.Ar(70,123.905895774),new n.Ar(71,124.906398236),new n.Ar(72,125.904268868),new n.Ar(73,126.905179581),new n.Ar(74,127.903530436),new n.Ar(75,128.904779458),new n.Ar(76,129.903507903),new n.Ar(77,130.90508192),new n.Ar(78,131.904154457),new n.Ar(79,132.90590566),new n.Ar(80,133.905394504),new n.Ar(81,134.907207499),new n.Ar(82,135.907219526),new n.Ar(83,136.911562939),new n.Ar(84,137.913988549),new n.Ar(85,138.918786859),new n.Ar(86,139.921635665),new n.Ar(87,140.926646282),new n.Ar(88,141.929702981),new n.Ar(89,142.93489),new n.Ar(90,143.93823),new n.Ar(91,144.94367),new n.Ar(92,145.9473),new n.Ar(93,146.95301)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(57,111.950331),new n.Ar(58,112.944535512),new n.Ar(59,113.940841319),new n.Ar(60,114.935939),new n.Ar(61,115.932914152),new n.Ar(62,116.928639484),new n.Ar(63,117.926554883),new n.Ar(64,118.922370879),new n.Ar(65,119.920678219),new n.Ar(66,120.917183637),new n.Ar(67,121.916121946),new n.Ar(68,122.912990168),new n.Ar(69,123.912245731),new n.Ar(70,124.909724871),new n.Ar(71,125.909447953),new n.Ar(72,126.9074176),new n.Ar(73,127.907747919),new n.Ar(74,128.906063369),new n.Ar(75,129.906706163),new n.Ar(76,130.905460232),new n.Ar(77,131.906429799),new n.Ar(78,132.90544687),new n.Ar(79,133.906713419),new n.Ar(80,134.905971903),new n.Ar(81,135.907305741),new n.Ar(82,136.907083505),new n.Ar(83,137.911010537),new n.Ar(84,138.913357921),new n.Ar(85,139.917277075),new n.Ar(86,140.920043984),new n.Ar(87,141.924292317),new n.Ar(88,142.927330292),new n.Ar(89,143.932027373),new n.Ar(90,144.935388226),new n.Ar(91,145.940162028),new n.Ar(92,146.943864435),new n.Ar(93,147.948899539),new n.Ar(94,148.95272),new n.Ar(95,149.95797),new n.Ar(96,150.962)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(58,113.950941),new n.Ar(59,114.94771),new n.Ar(60,115.94168),new n.Ar(61,116.937700229),new n.Ar(62,117.93344),new n.Ar(63,118.931051927),new n.Ar(64,119.926045941),new n.Ar(65,120.924485908),new n.Ar(66,121.92026),new n.Ar(67,122.91885),new n.Ar(68,123.915088437),new n.Ar(69,124.914620234),new n.Ar(70,125.911244146),new n.Ar(71,126.911121328),new n.Ar(72,127.90830887),new n.Ar(73,128.908673749),new n.Ar(74,129.906310478),new n.Ar(75,130.906930798),new n.Ar(76,131.905056152),new n.Ar(77,132.906002368),new n.Ar(78,133.904503347),new n.Ar(79,134.905682749),new n.Ar(80,135.904570109),new n.Ar(81,136.905821414),new n.Ar(82,137.905241273),new n.Ar(83,138.908835384),new n.Ar(84,139.910599485),new n.Ar(85,140.914406439),new n.Ar(86,141.916448175),new n.Ar(87,142.920617184),new n.Ar(88,143.922940468),new n.Ar(89,144.926923807),new n.Ar(90,145.930106645),new n.Ar(91,146.933992519),new n.Ar(92,147.937682377),new n.Ar(93,148.94246),new n.Ar(94,149.94562),new n.Ar(95,150.9507),new n.Ar(96,151.95416),new n.Ar(97,152.95961)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(60,116.95001),new n.Ar(61,117.94657),new n.Ar(62,118.94099),new n.Ar(63,119.93807),new n.Ar(64,120.93301),new n.Ar(65,121.93071),new n.Ar(66,122.92624),new n.Ar(67,123.92453),new n.Ar(68,124.92067),new n.Ar(69,125.91937),new n.Ar(70,126.91616),new n.Ar(71,127.91544794),new n.Ar(72,128.912667334),new n.Ar(73,129.91232),new n.Ar(74,130.910108489),new n.Ar(75,131.910110399),new n.Ar(76,132.908396372),new n.Ar(77,133.908489607),new n.Ar(78,134.906971003),new n.Ar(79,135.907651181),new n.Ar(80,136.906465656),new n.Ar(81,137.907106826),new n.Ar(82,138.90634816),new n.Ar(83,139.909472552),new n.Ar(84,140.910957016),new n.Ar(85,141.914074489),new n.Ar(86,142.916058646),new n.Ar(87,143.919591666),new n.Ar(88,144.92163837),new n.Ar(89,145.925700146),new n.Ar(90,146.927819639),new n.Ar(91,147.932191197),new n.Ar(92,148.93437),new n.Ar(93,149.93857),new n.Ar(94,150.94156),new n.Ar(95,151.94611),new n.Ar(96,152.94945),new n.Ar(97,153.9544),new n.Ar(98,154.95813)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(61,118.95276),new n.Ar(62,119.94664),new n.Ar(63,120.94367),new n.Ar(64,121.93801),new n.Ar(65,122.93551),new n.Ar(66,123.93052),new n.Ar(67,124.92854),new n.Ar(68,125.9241),new n.Ar(69,126.92275),new n.Ar(70,127.91887),new n.Ar(71,128.918679183),new n.Ar(72,129.914339361),new n.Ar(73,130.914424137),new n.Ar(74,131.91149),new n.Ar(75,132.91155),new n.Ar(76,133.909026379),new n.Ar(77,134.909145555),new n.Ar(78,135.907143574),new n.Ar(79,136.907777634),new n.Ar(80,137.905985574),new n.Ar(81,138.906646605),new n.Ar(82,139.905434035),new n.Ar(83,140.908271103),new n.Ar(84,141.909239733),new n.Ar(85,142.912381158),new n.Ar(86,143.913642686),new n.Ar(87,144.917227871),new n.Ar(88,145.918689722),new n.Ar(89,146.922510962),new n.Ar(90,147.924394738),new n.Ar(91,148.928289207),new n.Ar(92,149.930226399),new n.Ar(93,150.93404),new n.Ar(94,151.93638),new n.Ar(95,152.94058),new n.Ar(96,153.94332),new n.Ar(97,154.94804),new n.Ar(98,155.95126),new n.Ar(99,156.95634)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(62,120.955364),new n.Ar(63,121.95165),new n.Ar(64,122.94596),new n.Ar(65,123.94296),new n.Ar(66,124.93783),new n.Ar(67,125.93531),new n.Ar(68,126.93083),new n.Ar(69,127.9288),new n.Ar(70,128.92486),new n.Ar(71,129.92338),new n.Ar(72,130.920060245),new n.Ar(73,131.91912),new n.Ar(74,132.9162),new n.Ar(75,133.915672),new n.Ar(76,134.91313914),new n.Ar(77,135.912646935),new n.Ar(78,136.910678351),new n.Ar(79,137.910748891),new n.Ar(80,138.908932181),new n.Ar(81,139.909071204),new n.Ar(82,140.907647726),new n.Ar(83,141.910039865),new n.Ar(84,142.910812233),new n.Ar(85,143.913300595),new n.Ar(86,144.914506897),new n.Ar(87,145.917588016),new n.Ar(88,146.918979001),new n.Ar(89,147.922183237),new n.Ar(90,148.923791056),new n.Ar(91,149.926995031),new n.Ar(92,150.928227869),new n.Ar(93,151.9316),new n.Ar(94,152.93365),new n.Ar(95,153.93739),new n.Ar(96,154.93999),new n.Ar(97,155.94412),new n.Ar(98,156.94717),new n.Ar(99,157.95178),new n.Ar(100,158.95523)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(66,125.94307),new n.Ar(67,126.9405),new n.Ar(68,127.93539),new n.Ar(69,128.932385),new n.Ar(70,129.92878),new n.Ar(71,130.927102697),new n.Ar(72,131.92312),new n.Ar(73,132.92221),new n.Ar(74,133.918645),new n.Ar(75,134.91824),new n.Ar(76,135.915020542),new n.Ar(77,136.91463973),new n.Ar(78,137.91291745),new n.Ar(79,138.91192415),new n.Ar(80,139.909309824),new n.Ar(81,140.9096048),new n.Ar(82,141.907718643),new n.Ar(83,142.909809626),new n.Ar(84,143.910082629),new n.Ar(85,144.912568847),new n.Ar(86,145.913112139),new n.Ar(87,146.916095794),new n.Ar(88,147.916888516),new n.Ar(89,148.92014419),new n.Ar(90,149.920886563),new n.Ar(91,150.923824739),new n.Ar(92,151.924682428),new n.Ar(93,152.927694534),new n.Ar(94,153.929483295),new n.Ar(95,154.932629551),new n.Ar(96,155.9352),new n.Ar(97,156.93927),new n.Ar(98,157.94187),new n.Ar(99,158.94639),new n.Ar(100,159.94939),new n.Ar(101,160.95433)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(67,127.94826),new n.Ar(68,128.94316),new n.Ar(69,129.94045),new n.Ar(70,130.9358),new n.Ar(71,131.93375),new n.Ar(72,132.92972),new n.Ar(73,133.92849),new n.Ar(74,134.924617),new n.Ar(75,135.923447865),new n.Ar(76,136.920713),new n.Ar(77,137.920432261),new n.Ar(78,138.916759814),new n.Ar(79,139.915801649),new n.Ar(80,140.913606636),new n.Ar(81,141.912950738),new n.Ar(82,142.910927571),new n.Ar(83,143.912585768),new n.Ar(84,144.912743879),new n.Ar(85,145.914692165),new n.Ar(86,146.915133898),new n.Ar(87,147.917467786),new n.Ar(88,148.918329195),new n.Ar(89,149.920979477),new n.Ar(90,150.921202693),new n.Ar(91,151.923490557),new n.Ar(92,152.924113189),new n.Ar(93,153.926547019),new n.Ar(94,154.928097047),new n.Ar(95,155.931060357),new n.Ar(96,156.9332),new n.Ar(97,157.93669),new n.Ar(98,158.93913),new n.Ar(99,159.94299),new n.Ar(100,160.94586),new n.Ar(101,161.95029),new n.Ar(102,162.95352)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(68,129.94863),new n.Ar(69,130.94589),new n.Ar(70,131.94082),new n.Ar(71,132.93873),new n.Ar(72,133.93402),new n.Ar(73,134.93235),new n.Ar(74,135.9283),new n.Ar(75,136.927046709),new n.Ar(76,137.92354),new n.Ar(77,138.922302),new n.Ar(78,139.918991),new n.Ar(79,140.918468512),new n.Ar(80,141.915193274),new n.Ar(81,142.914623555),new n.Ar(82,143.91199473),new n.Ar(83,144.913405611),new n.Ar(84,145.91303676),new n.Ar(85,146.914893275),new n.Ar(86,147.914817914),new n.Ar(87,148.917179521),new n.Ar(88,149.917271454),new n.Ar(89,150.919928351),new n.Ar(90,151.919728244),new n.Ar(91,152.922093907),new n.Ar(92,153.922205303),new n.Ar(93,154.92463594),new n.Ar(94,155.925526236),new n.Ar(95,156.928354506),new n.Ar(96,157.929987938),new n.Ar(97,158.9332),new n.Ar(98,159.93514),new n.Ar(99,160.93883),new n.Ar(100,161.94122),new n.Ar(101,162.94536),new n.Ar(102,163.94828),new n.Ar(103,164.95298)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(69,131.95416),new n.Ar(70,132.9489),new n.Ar(71,133.94632),new n.Ar(72,134.94172),new n.Ar(73,135.9395),new n.Ar(74,136.93521),new n.Ar(75,137.93345),new n.Ar(76,138.92882915),new n.Ar(77,139.928083921),new n.Ar(78,140.924885867),new n.Ar(79,141.923400033),new n.Ar(80,142.920286634),new n.Ar(81,143.918774116),new n.Ar(82,144.916261285),new n.Ar(83,145.917199714),new n.Ar(84,146.916741206),new n.Ar(85,147.918153775),new n.Ar(86,148.917925922),new n.Ar(87,149.919698294),new n.Ar(88,150.919846022),new n.Ar(89,151.921740399),new n.Ar(90,152.921226219),new n.Ar(91,153.922975386),new n.Ar(92,154.922889429),new n.Ar(93,155.924750855),new n.Ar(94,156.925419435),new n.Ar(95,157.927841923),new n.Ar(96,158.9290845),new n.Ar(97,159.931460406),new n.Ar(98,160.93368),new n.Ar(99,161.93704),new n.Ar(100,162.93921),new n.Ar(101,163.94299),new n.Ar(102,164.94572),new n.Ar(103,165.94997),new n.Ar(104,166.95305)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(72,135.94707),new n.Ar(73,136.94465),new n.Ar(74,137.93997),new n.Ar(75,138.93808),new n.Ar(76,139.933236934),new n.Ar(77,140.93221),new n.Ar(78,141.927908919),new n.Ar(79,142.926738636),new n.Ar(80,143.923390357),new n.Ar(81,144.921687498),new n.Ar(82,145.918305344),new n.Ar(83,146.919089446),new n.Ar(84,147.918109771),new n.Ar(85,148.919336427),new n.Ar(86,149.918655455),new n.Ar(87,150.920344273),new n.Ar(88,151.919787882),new n.Ar(89,152.921746283),new n.Ar(90,153.920862271),new n.Ar(91,154.922618801),new n.Ar(92,155.922119552),new n.Ar(93,156.923956686),new n.Ar(94,157.924100533),new n.Ar(95,158.926385075),new n.Ar(96,159.927050616),new n.Ar(97,160.929665688),new n.Ar(98,161.930981211),new n.Ar(99,162.93399),new n.Ar(100,163.93586),new n.Ar(101,164.93938),new n.Ar(102,165.9416),new n.Ar(103,166.94557),new n.Ar(104,167.94836),new n.Ar(105,168.95287)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(73,137.95287),new n.Ar(74,138.94803),new n.Ar(75,139.945367985),new n.Ar(76,140.94116),new n.Ar(77,141.939073781),new n.Ar(78,142.93475),new n.Ar(79,143.93253),new n.Ar(80,144.92888),new n.Ar(81,145.927180629),new n.Ar(82,146.924037176),new n.Ar(83,147.924298636),new n.Ar(84,148.92324163),new n.Ar(85,149.923654158),new n.Ar(86,150.923098169),new n.Ar(87,151.924071324),new n.Ar(88,152.923430858),new n.Ar(89,153.924686236),new n.Ar(90,154.923500411),new n.Ar(91,155.924743749),new n.Ar(92,156.924021155),new n.Ar(93,157.92541026),new n.Ar(94,158.925343135),new n.Ar(95,159.927164021),new n.Ar(96,160.927566289),new n.Ar(97,161.929484803),new n.Ar(98,162.930643942),new n.Ar(99,163.933347253),new n.Ar(100,164.93488),new n.Ar(101,165.93805),new n.Ar(102,166.94005),new n.Ar(103,167.94364),new n.Ar(104,168.94622),new n.Ar(105,169.95025),new n.Ar(106,170.9533)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(74,139.95379),new n.Ar(75,140.95119),new n.Ar(76,141.946695946),new n.Ar(77,142.94383),new n.Ar(78,143.93907),new n.Ar(79,144.936717),new n.Ar(80,145.932720118),new n.Ar(81,146.930878496),new n.Ar(82,147.927177882),new n.Ar(83,148.927333981),new n.Ar(84,149.925579728),new n.Ar(85,150.92617963),new n.Ar(86,151.924713874),new n.Ar(87,152.925760865),new n.Ar(88,153.924422046),new n.Ar(89,154.92574895),new n.Ar(90,155.924278273),new n.Ar(91,156.925461256),new n.Ar(92,157.924404637),new n.Ar(93,158.92573566),new n.Ar(94,159.925193718),new n.Ar(95,160.926929595),new n.Ar(96,161.926794731),new n.Ar(97,162.928727532),new n.Ar(98,163.929171165),new n.Ar(99,164.931699828),new n.Ar(100,165.932803241),new n.Ar(101,166.935649025),new n.Ar(102,167.93723),new n.Ar(103,168.940303648),new n.Ar(104,169.94267),new n.Ar(105,170.94648),new n.Ar(106,171.94911),new n.Ar(107,172.95344)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(75,141.95986),new n.Ar(76,142.95469),new n.Ar(77,143.95164),new n.Ar(78,144.94688),new n.Ar(79,145.9441),new n.Ar(80,146.93984),new n.Ar(81,147.937269),new n.Ar(82,148.933789944),new n.Ar(83,149.932760914),new n.Ar(84,150.931680791),new n.Ar(85,151.931740598),new n.Ar(86,152.930194506),new n.Ar(87,153.930596268),new n.Ar(88,154.929079084),new n.Ar(89,155.929001869),new n.Ar(90,156.928188059),new n.Ar(91,157.92894573),new n.Ar(92,158.927708537),new n.Ar(93,159.928725679),new n.Ar(94,160.927851662),new n.Ar(95,161.92909242),new n.Ar(96,162.928730286),new n.Ar(97,163.930230577),new n.Ar(98,164.930319169),new n.Ar(99,165.932281267),new n.Ar(100,166.933126195),new n.Ar(101,167.935496424),new n.Ar(102,168.936868306),new n.Ar(103,169.939614951),new n.Ar(104,170.941461227),new n.Ar(105,171.94482),new n.Ar(106,172.94729),new n.Ar(107,173.95115),new n.Ar(108,174.95405)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(76,143.96059),new n.Ar(77,144.95746),new n.Ar(78,145.95212),new n.Ar(79,146.94931),new n.Ar(80,147.94444),new n.Ar(81,148.942780527),new n.Ar(82,149.937171034),new n.Ar(83,150.93746),new n.Ar(84,151.935078452),new n.Ar(85,152.935093125),new n.Ar(86,153.932777294),new n.Ar(87,154.933204273),new n.Ar(88,155.931015001),new n.Ar(89,156.931945517),new n.Ar(90,157.929912),new n.Ar(91,158.930680718),new n.Ar(92,159.929078924),new n.Ar(93,160.930001348),new n.Ar(94,161.928774923),new n.Ar(95,162.930029273),new n.Ar(96,163.929196996),new n.Ar(97,164.9307228),new n.Ar(98,165.93028997),new n.Ar(99,166.932045448),new n.Ar(100,167.932367781),new n.Ar(101,168.934588082),new n.Ar(102,169.935460334),new n.Ar(103,170.938025885),new n.Ar(104,171.939352149),new n.Ar(105,172.9424),new n.Ar(106,173.94434),new n.Ar(107,174.94793),new n.Ar(108,175.95029),new n.Ar(109,176.95437)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(77,145.966495),new n.Ar(78,146.961081),new n.Ar(79,147.95755),new n.Ar(80,148.95265),new n.Ar(81,149.94967),new n.Ar(82,150.944842),new n.Ar(83,151.9443),new n.Ar(84,152.942027631),new n.Ar(85,153.940832325),new n.Ar(86,154.939191562),new n.Ar(87,155.939006895),new n.Ar(88,156.936756069),new n.Ar(89,157.936996),new n.Ar(90,158.934808966),new n.Ar(91,159.935090772),new n.Ar(92,160.933398042),new n.Ar(93,161.933970147),new n.Ar(94,162.932647648),new n.Ar(95,163.933450972),new n.Ar(96,164.932432463),new n.Ar(97,165.933553133),new n.Ar(98,166.932848844),new n.Ar(99,167.934170375),new n.Ar(100,168.934211117),new n.Ar(101,169.935797877),new n.Ar(102,170.936425817),new n.Ar(103,171.938396118),new n.Ar(104,172.939600336),new n.Ar(105,173.942164618),new n.Ar(106,174.943832897),new n.Ar(107,175.946991412),new n.Ar(108,176.94904),new n.Ar(109,177.95264),new n.Ar(110,178.95534)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(78,147.96676),new n.Ar(79,148.96348),new n.Ar(80,149.95799),new n.Ar(81,150.954657965),new n.Ar(82,151.950167),new n.Ar(83,152.94921),new n.Ar(84,153.945651145),new n.Ar(85,154.945792),new n.Ar(86,155.942847109),new n.Ar(87,156.94265865),new n.Ar(88,157.939857897),new n.Ar(89,158.940153735),new n.Ar(90,159.93756),new n.Ar(91,160.937357719),new n.Ar(92,161.93575),new n.Ar(93,162.936265492),new n.Ar(94,163.93452),new n.Ar(95,164.935397592),new n.Ar(96,165.933879623),new n.Ar(97,166.934946862),new n.Ar(98,167.933894465),new n.Ar(99,168.93518712),new n.Ar(100,169.934758652),new n.Ar(101,170.936322297),new n.Ar(102,171.936377696),new n.Ar(103,172.938206756),new n.Ar(104,173.938858101),new n.Ar(105,174.941272494),new n.Ar(106,175.942568409),new n.Ar(107,176.945257126),new n.Ar(108,177.946643396),new n.Ar(109,178.95017),new n.Ar(110,179.95233),new n.Ar(111,180.95615)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(79,149.972668),new n.Ar(80,150.967147),new n.Ar(81,151.96361),new n.Ar(82,152.95869),new n.Ar(83,153.9571),new n.Ar(84,154.953641324),new n.Ar(85,155.952907),new n.Ar(86,156.950101536),new n.Ar(87,157.948577981),new n.Ar(88,158.946615113),new n.Ar(89,159.945383),new n.Ar(90,160.943047504),new n.Ar(91,161.943222),new n.Ar(92,162.941203796),new n.Ar(93,163.941215),new n.Ar(94,164.939605886),new n.Ar(95,165.939762646),new n.Ar(96,166.938307056),new n.Ar(97,167.938698576),new n.Ar(98,168.937648757),new n.Ar(99,169.93847219),new n.Ar(100,170.937909903),new n.Ar(101,171.939082239),new n.Ar(102,172.938926901),new n.Ar(103,173.940333522),new n.Ar(104,174.940767904),new n.Ar(105,175.942682399),new n.Ar(106,176.943754987),new n.Ar(107,177.945951366),new n.Ar(108,178.947324216),new n.Ar(109,179.949879968),new n.Ar(110,180.95197),new n.Ar(111,181.95521),new n.Ar(112,182.95757),new n.Ar(113,183.96117)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(82,153.96425),new n.Ar(83,154.96276),new n.Ar(84,155.959247),new n.Ar(85,156.958127),new n.Ar(86,157.95405528),new n.Ar(87,158.954003),new n.Ar(88,159.950713588),new n.Ar(89,160.950330852),new n.Ar(90,161.947202977),new n.Ar(91,162.947057),new n.Ar(92,163.944422),new n.Ar(93,164.94454),new n.Ar(94,165.94225),new n.Ar(95,166.9426),new n.Ar(96,167.94063),new n.Ar(97,168.941158567),new n.Ar(98,169.93965),new n.Ar(99,170.94049),new n.Ar(100,171.93945798),new n.Ar(101,172.94065),new n.Ar(102,173.940040159),new n.Ar(103,174.941502991),new n.Ar(104,175.941401828),new n.Ar(105,176.943220013),new n.Ar(106,177.943697732),new n.Ar(107,178.945815073),new n.Ar(108,179.94654876),new n.Ar(109,180.949099124),new n.Ar(110,181.950552893),new n.Ar(111,182.953531012),new n.Ar(112,183.95544788),new n.Ar(113,184.95878),new n.Ar(114,185.96092)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(83,155.971689),new n.Ar(84,156.968145),new n.Ar(85,157.966368),new n.Ar(86,158.96232309),new n.Ar(87,159.961358),new n.Ar(88,160.958372992),new n.Ar(89,161.956556553),new n.Ar(90,162.95431665),new n.Ar(91,163.95357),new n.Ar(92,164.950817),new n.Ar(93,165.95047),new n.Ar(94,166.948639),new n.Ar(95,167.947787),new n.Ar(96,168.94592),new n.Ar(97,169.94609),new n.Ar(98,170.94446),new n.Ar(99,171.944739818),new n.Ar(100,172.94459),new n.Ar(101,173.944167937),new n.Ar(102,174.94365),new n.Ar(103,175.944740551),new n.Ar(104,176.944471766),new n.Ar(105,177.945750349),new n.Ar(106,178.945934113),new n.Ar(107,179.947465655),new n.Ar(108,180.947996346),new n.Ar(109,181.950152414),new n.Ar(110,182.951373188),new n.Ar(111,183.954009331),new n.Ar(112,184.955559086),new n.Ar(113,185.9585501),new n.Ar(114,186.96041),new n.Ar(115,187.96371)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(84,157.973939),new n.Ar(85,158.97228),new n.Ar(86,159.968369),new n.Ar(87,160.967089),new n.Ar(88,161.962750303),new n.Ar(89,162.962532),new n.Ar(90,163.95898381),new n.Ar(91,164.958335962),new n.Ar(92,165.955019896),new n.Ar(93,166.954672),new n.Ar(94,167.951863),new n.Ar(95,168.951759),new n.Ar(96,169.948473988),new n.Ar(97,170.94946),new n.Ar(98,171.948228837),new n.Ar(99,172.948884),new n.Ar(100,173.94616),new n.Ar(101,174.94677),new n.Ar(102,175.94559),new n.Ar(103,176.94662),new n.Ar(104,177.945848364),new n.Ar(105,178.947071733),new n.Ar(106,179.946705734),new n.Ar(107,180.948198054),new n.Ar(108,181.948205519),new n.Ar(109,182.950224458),new n.Ar(110,183.950932553),new n.Ar(111,184.953420586),new n.Ar(112,185.954362204),new n.Ar(113,186.957158365),new n.Ar(114,187.958486954),new n.Ar(115,188.96191222),new n.Ar(116,189.963179541)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(85,159.981485),new n.Ar(86,160.977661),new n.Ar(87,161.975707),new n.Ar(88,162.971375872),new n.Ar(89,163.970319),new n.Ar(90,164.967050268),new n.Ar(91,165.965211372),new n.Ar(92,166.962564),new n.Ar(93,167.961609),new n.Ar(94,168.95883),new n.Ar(95,169.958163),new n.Ar(96,170.955547),new n.Ar(97,171.955285),new n.Ar(98,172.953062),new n.Ar(99,173.952114),new n.Ar(100,174.951393),new n.Ar(101,175.95157),new n.Ar(102,176.95027),new n.Ar(103,177.950851081),new n.Ar(104,178.949981038),new n.Ar(105,179.95078768),new n.Ar(106,180.950064596),new n.Ar(107,181.951211444),new n.Ar(108,182.950821349),new n.Ar(109,183.952524289),new n.Ar(110,184.952955747),new n.Ar(111,185.954986529),new n.Ar(112,186.955750787),new n.Ar(113,187.958112287),new n.Ar(114,188.959228359),new n.Ar(115,189.961816139),new n.Ar(116,190.963123592),new n.Ar(117,191.96596)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(86,161.983819),new n.Ar(87,162.982048),new n.Ar(88,163.977927),new n.Ar(89,164.976475),new n.Ar(90,165.971934911),new n.Ar(91,166.971554),new n.Ar(92,167.967832911),new n.Ar(93,168.967076205),new n.Ar(94,169.963569716),new n.Ar(95,170.96304),new n.Ar(96,171.960078),new n.Ar(97,172.959791),new n.Ar(98,173.956307704),new n.Ar(99,174.95708),new n.Ar(100,175.953757941),new n.Ar(101,176.955045),new n.Ar(102,177.953348225),new n.Ar(103,178.953951),new n.Ar(104,179.952308241),new n.Ar(105,180.953274494),new n.Ar(106,181.952186222),new n.Ar(107,182.95311),new n.Ar(108,183.952490808),new n.Ar(109,184.954043023),new n.Ar(110,185.953838355),new n.Ar(111,186.955747928),new n.Ar(112,187.955835993),new n.Ar(113,188.958144866),new n.Ar(114,189.95844521),new n.Ar(115,190.960927951),new n.Ar(116,191.961479047),new n.Ar(117,192.964148083),new n.Ar(118,193.965179314),new n.Ar(119,194.968123889),new n.Ar(120,195.96962255)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(88,164.98758),new n.Ar(89,165.985506),new n.Ar(90,166.980951577),new n.Ar(91,167.979966),new n.Ar(92,168.976390868),new n.Ar(93,169.974441697),new n.Ar(94,170.971779),new n.Ar(95,171.970643),new n.Ar(96,172.967707),new n.Ar(97,173.966804),new n.Ar(98,174.964279),new n.Ar(99,175.963511),new n.Ar(100,176.96117),new n.Ar(101,177.960084944),new n.Ar(102,178.95915),new n.Ar(103,179.958555615),new n.Ar(104,180.957642156),new n.Ar(105,181.958127689),new n.Ar(106,182.956814),new n.Ar(107,183.957388318),new n.Ar(108,184.95659),new n.Ar(109,185.957951104),new n.Ar(110,186.95736083),new n.Ar(111,187.958851962),new n.Ar(112,188.958716473),new n.Ar(113,189.960592299),new n.Ar(114,190.960591191),new n.Ar(115,191.962602198),new n.Ar(116,192.9629237),new n.Ar(117,193.96507561),new n.Ar(118,194.9659768),new n.Ar(119,195.968379906),new n.Ar(120,196.969636496),new n.Ar(121,197.97228),new n.Ar(122,198.973787159)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(90,167.988035),new n.Ar(91,168.986421),new n.Ar(92,169.981734918),new n.Ar(93,170.981251),new n.Ar(94,171.977376138),new n.Ar(95,172.976499642),new n.Ar(96,173.972811276),new n.Ar(97,174.972276),new n.Ar(98,175.969),new n.Ar(99,176.968453),new n.Ar(100,177.964894223),new n.Ar(101,178.965475),new n.Ar(102,179.962023729),new n.Ar(103,180.963177),new n.Ar(104,181.961267637),new n.Ar(105,182.961729),new n.Ar(106,183.959851685),new n.Ar(107,184.960753782),new n.Ar(108,185.959432346),new n.Ar(109,186.960697),new n.Ar(110,187.959395697),new n.Ar(111,188.9608319),new n.Ar(112,189.959930073),new n.Ar(113,190.961684653),new n.Ar(114,191.961035158),new n.Ar(115,192.962984504),new n.Ar(116,193.962663581),new n.Ar(117,194.964774449),new n.Ar(118,195.964934884),new n.Ar(119,196.967323401),new n.Ar(120,197.967876009),new n.Ar(121,198.970576213),new n.Ar(122,199.971423885),new n.Ar(123,200.974496467),new n.Ar(124,201.97574)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(92,170.991183),new n.Ar(93,171.990109),new n.Ar(94,172.986398138),new n.Ar(95,173.984325861),new n.Ar(96,174.981552),new n.Ar(97,175.980269),new n.Ar(98,176.977215),new n.Ar(99,177.975975),new n.Ar(100,178.973412),new n.Ar(101,179.972396),new n.Ar(102,180.969948),new n.Ar(103,181.968621416),new n.Ar(104,182.96762),new n.Ar(105,183.966776046),new n.Ar(106,184.965806956),new n.Ar(107,185.965997671),new n.Ar(108,186.964562),new n.Ar(109,187.965321662),new n.Ar(110,188.9642243),new n.Ar(111,189.964698757),new n.Ar(112,190.963649239),new n.Ar(113,191.964810107),new n.Ar(114,192.964131745),new n.Ar(115,193.96533889),new n.Ar(116,194.965017928),new n.Ar(117,195.966551315),new n.Ar(118,196.966551609),new n.Ar(119,197.968225244),new n.Ar(120,198.968748016),new n.Ar(121,199.970717886),new n.Ar(122,200.971640839),new n.Ar(123,201.973788431),new n.Ar(124,202.975137256),new n.Ar(125,203.977705),new n.Ar(126,204.97961)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(95,174.991411),new n.Ar(96,175.987413248),new n.Ar(97,176.986336874),new n.Ar(98,177.982476325),new n.Ar(99,178.981783),new n.Ar(100,179.978322),new n.Ar(101,180.977806),new n.Ar(102,181.97393546),new n.Ar(103,182.974561),new n.Ar(104,183.970705219),new n.Ar(105,184.971983),new n.Ar(106,185.969460021),new n.Ar(107,186.969785),new n.Ar(108,187.967511693),new n.Ar(109,188.968733187),new n.Ar(110,189.966958568),new n.Ar(111,190.96706311),new n.Ar(112,191.965921572),new n.Ar(113,192.966644169),new n.Ar(114,193.965381832),new n.Ar(115,194.966638981),new n.Ar(116,195.965814846),new n.Ar(117,196.967195333),new n.Ar(118,197.96675183),new n.Ar(119,198.968262489),new n.Ar(120,199.968308726),new n.Ar(121,200.970285275),new n.Ar(122,201.970625604),new n.Ar(123,202.972857096),new n.Ar(124,203.97347564),new n.Ar(125,204.976056104),new n.Ar(126,205.977498672),new n.Ar(127,206.982577025),new n.Ar(128,207.98594)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(96,176.996881),new n.Ar(97,177.994637),new n.Ar(98,178.991466),new n.Ar(99,179.990194),new n.Ar(100,180.986904),new n.Ar(101,181.98561),new n.Ar(102,182.982697),new n.Ar(103,183.98176),new n.Ar(104,184.9791),new n.Ar(105,185.977549881),new n.Ar(106,186.97617),new n.Ar(107,187.97592),new n.Ar(108,188.974290451),new n.Ar(109,189.974473379),new n.Ar(110,190.972261952),new n.Ar(111,191.972770785),new n.Ar(112,192.970548),new n.Ar(113,193.971053),new n.Ar(114,194.96965),new n.Ar(115,195.970515),new n.Ar(116,196.9695362),new n.Ar(117,197.970466294),new n.Ar(118,198.969813837),new n.Ar(119,199.970945394),new n.Ar(120,200.97080377),new n.Ar(121,201.972090569),new n.Ar(122,202.972329088),new n.Ar(123,203.973848646),new n.Ar(124,204.97441227),new n.Ar(125,205.976095321),new n.Ar(126,206.977407908),new n.Ar(127,207.982004653),new n.Ar(128,208.985349125),new n.Ar(129,209.990065574)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(99,180.996714),new n.Ar(100,181.992676101),new n.Ar(101,182.99193),new n.Ar(102,183.988198),new n.Ar(103,184.98758),new n.Ar(104,185.983485388),new n.Ar(105,186.98403),new n.Ar(106,187.979869108),new n.Ar(107,188.98088),new n.Ar(108,189.978180008),new n.Ar(109,190.9782),new n.Ar(110,191.975719811),new n.Ar(111,192.97608),new n.Ar(112,193.974648056),new n.Ar(113,194.975920279),new n.Ar(114,195.97271),new n.Ar(115,196.97338),new n.Ar(116,197.97198),new n.Ar(117,198.972909384),new n.Ar(118,199.97181556),new n.Ar(119,200.972846589),new n.Ar(120,201.972143786),new n.Ar(121,202.973375491),new n.Ar(122,203.973028761),new n.Ar(123,204.974467112),new n.Ar(124,205.974449002),new n.Ar(125,206.975880605),new n.Ar(126,207.97663585),new n.Ar(127,208.981074801),new n.Ar(128,209.984173129),new n.Ar(129,210.988731474),new n.Ar(130,211.991887495),new n.Ar(131,212.9965),new n.Ar(132,213.999798147)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(102,184.997708),new n.Ar(103,185.99648),new n.Ar(104,186.993458),new n.Ar(105,187.992173),new n.Ar(106,188.989505),new n.Ar(107,189.987520007),new n.Ar(108,190.986053),new n.Ar(109,191.985368),new n.Ar(110,192.983662229),new n.Ar(111,193.983430186),new n.Ar(112,194.98112697),new n.Ar(113,195.981236107),new n.Ar(114,196.978934287),new n.Ar(115,197.979024396),new n.Ar(116,198.977576953),new n.Ar(117,199.978141983),new n.Ar(118,200.976970721),new n.Ar(119,201.977674504),new n.Ar(120,202.976868118),new n.Ar(121,203.977805161),new n.Ar(122,204.977374688),new n.Ar(123,205.978482854),new n.Ar(124,206.978455217),new n.Ar(125,207.979726699),new n.Ar(126,208.980383241),new n.Ar(127,209.984104944),new n.Ar(128,210.987258139),new n.Ar(129,211.991271542),new n.Ar(130,212.994374836),new n.Ar(131,213.998698664),new n.Ar(132,215.001832349),new n.Ar(133,216.006199)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(106,189.994293888),new n.Ar(107,190.994653),new n.Ar(108,191.99033039),new n.Ar(109,192.991102),new n.Ar(110,193.988284107),new n.Ar(111,194.988045),new n.Ar(112,195.985469432),new n.Ar(113,196.985567),new n.Ar(114,197.984024384),new n.Ar(115,198.985044507),new n.Ar(116,199.981735),new n.Ar(117,200.982209),new n.Ar(118,201.980704),new n.Ar(119,202.981412863),new n.Ar(120,203.980307113),new n.Ar(121,204.981165396),new n.Ar(122,205.980465241),new n.Ar(123,206.981578228),new n.Ar(124,207.981231059),new n.Ar(125,208.982415788),new n.Ar(126,209.982857396),new n.Ar(127,210.986636869),new n.Ar(128,211.988851755),new n.Ar(129,212.992842522),new n.Ar(130,213.995185949),new n.Ar(131,214.999414609),new n.Ar(132,216.001905198),new n.Ar(133,217.006253),new n.Ar(134,218.008965773)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(108,193.000188),new n.Ar(109,193.997973),new n.Ar(110,194.996554),new n.Ar(111,195.995702),new n.Ar(112,196.993891293),new n.Ar(113,197.99343368),new n.Ar(114,198.991008569),new n.Ar(115,199.990920883),new n.Ar(116,200.988486908),new n.Ar(117,201.988448629),new n.Ar(118,202.986847216),new n.Ar(119,203.987261559),new n.Ar(120,204.986036352),new n.Ar(121,205.986599242),new n.Ar(122,206.985775861),new n.Ar(123,207.986582508),new n.Ar(124,208.986158678),new n.Ar(125,209.987131308),new n.Ar(126,210.987480806),new n.Ar(127,211.990734657),new n.Ar(128,212.99292115),new n.Ar(129,213.996356412),new n.Ar(130,214.998641245),new n.Ar(131,216.002408839),new n.Ar(132,217.004709619),new n.Ar(133,218.008681458),new n.Ar(134,219.011296478),new n.Ar(135,220.015301),new n.Ar(136,221.01814),new n.Ar(137,222.02233),new n.Ar(138,223.02534)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(110,196.001117268),new n.Ar(111,197.001661),new n.Ar(112,197.998779978),new n.Ar(113,198.998309),new n.Ar(114,199.995634148),new n.Ar(115,200.995535),new n.Ar(116,201.993899382),new n.Ar(117,202.994765192),new n.Ar(118,203.991365),new n.Ar(119,204.991668),new n.Ar(120,205.99016),new n.Ar(121,206.990726826),new n.Ar(122,207.989631237),new n.Ar(123,208.990376634),new n.Ar(124,209.989679862),new n.Ar(125,210.99058541),new n.Ar(126,211.990688899),new n.Ar(127,212.993868354),new n.Ar(128,213.995346275),new n.Ar(129,214.998729195),new n.Ar(130,216.000258153),new n.Ar(131,217.003914555),new n.Ar(132,218.005586315),new n.Ar(133,219.009474831),new n.Ar(134,220.011384149),new n.Ar(135,221.015455),new n.Ar(136,222.017570472),new n.Ar(137,223.02179),new n.Ar(138,224.02409),new n.Ar(139,225.02844),new n.Ar(140,226.03089),new n.Ar(141,227.035407),new n.Ar(142,228.038084)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(113,200.006499),new n.Ar(114,201.00458692),new n.Ar(115,202.00396885),new n.Ar(116,203.001423829),new n.Ar(117,204.001221209),new n.Ar(118,204.998663961),new n.Ar(119,205.998486886),new n.Ar(120,206.996859385),new n.Ar(121,207.997133849),new n.Ar(122,208.995915421),new n.Ar(123,209.996398327),new n.Ar(124,210.995529332),new n.Ar(125,211.996194988),new n.Ar(126,212.996174845),new n.Ar(127,213.99895474),new n.Ar(128,215.000326029),new n.Ar(129,216.003187873),new n.Ar(130,217.004616452),new n.Ar(131,218.007563326),new n.Ar(132,219.009240843),new n.Ar(133,220.012312978),new n.Ar(134,221.014245654),new n.Ar(135,222.017543957),new n.Ar(136,223.019730712),new n.Ar(137,224.023235513),new n.Ar(138,225.025606914),new n.Ar(139,226.029343423),new n.Ar(140,227.031833167),new n.Ar(141,228.034776087),new n.Ar(142,229.038426),new n.Ar(143,230.04251),new n.Ar(144,231.045407),new n.Ar(145,232.049654)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(115,203.00921),new n.Ar(116,204.006434513),new n.Ar(117,205.006187),new n.Ar(118,206.004463814),new n.Ar(119,207.005176607),new n.Ar(120,208.001776),new n.Ar(121,209.001944),new n.Ar(122,210.000446),new n.Ar(123,211.000893996),new n.Ar(124,211.999783492),new n.Ar(125,213.000345847),new n.Ar(126,214.000091141),new n.Ar(127,215.002704195),new n.Ar(128,216.003518402),new n.Ar(129,217.00630601),new n.Ar(130,218.007123948),new n.Ar(131,219.010068787),new n.Ar(132,220.011014669),new n.Ar(133,221.013907762),new n.Ar(134,222.01536182),new n.Ar(135,223.01849714),new n.Ar(136,224.020202004),new n.Ar(137,225.023604463),new n.Ar(138,226.025402555),new n.Ar(139,227.029170677),new n.Ar(140,228.031064101),new n.Ar(141,229.034820309),new n.Ar(142,230.037084774),new n.Ar(143,231.04122),new n.Ar(144,232.043693),new n.Ar(145,233.047995),new n.Ar(146,234.050547)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(118,207.012469754),new n.Ar(119,208.012112949),new n.Ar(120,209.009568736),new n.Ar(121,210.009256802),new n.Ar(122,211.007648196),new n.Ar(123,212.007811441),new n.Ar(124,213.006573689),new n.Ar(125,214.006893072),new n.Ar(126,215.006450832),new n.Ar(127,216.008721268),new n.Ar(128,217.009332676),new n.Ar(129,218.011625045),new n.Ar(130,219.012404918),new n.Ar(131,220.014752105),new n.Ar(132,221.015575746),new n.Ar(133,222.017828852),new n.Ar(134,223.01912603),new n.Ar(135,224.021708435),new n.Ar(136,225.023220576),new n.Ar(137,226.026089848),new n.Ar(138,227.027746979),new n.Ar(139,228.031014825),new n.Ar(140,229.032930871),new n.Ar(141,230.036025144),new n.Ar(142,231.038551503),new n.Ar(143,232.042022474),new n.Ar(144,233.04455),new n.Ar(145,234.04842),new n.Ar(146,235.051102),new n.Ar(147,236.055178)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(120,210.015711883),new n.Ar(121,211.016306912),new n.Ar(122,212.012916),new n.Ar(123,213.012962),new n.Ar(124,214.011451),new n.Ar(125,215.011726597),new n.Ar(126,216.011050963),new n.Ar(127,217.013066169),new n.Ar(128,218.013267744),new n.Ar(129,219.015521253),new n.Ar(130,220.015733126),new n.Ar(131,221.018171499),new n.Ar(132,222.018454131),new n.Ar(133,223.020795153),new n.Ar(134,224.02145925),new n.Ar(135,225.023941441),new n.Ar(136,226.024890681),new n.Ar(137,227.027698859),new n.Ar(138,228.028731348),new n.Ar(139,229.03175534),new n.Ar(140,230.033126574),new n.Ar(141,231.03629706),new n.Ar(142,232.03805036),new n.Ar(143,233.041576923),new n.Ar(144,234.043595497),new n.Ar(145,235.04750442),new n.Ar(146,236.04971),new n.Ar(147,237.053894),new n.Ar(148,238.056243)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(122,213.021183209),new n.Ar(123,214.02073923),new n.Ar(124,215.019097612),new n.Ar(125,216.019109649),new n.Ar(126,217.018288571),new n.Ar(127,218.020007906),new n.Ar(128,219.019880348),new n.Ar(129,220.021876493),new n.Ar(130,221.021863742),new n.Ar(131,222.023726),new n.Ar(132,223.023963748),new n.Ar(133,224.025614854),new n.Ar(134,225.026115172),new n.Ar(135,226.02793275),new n.Ar(136,227.028793151),new n.Ar(137,228.031036942),new n.Ar(138,229.032088601),new n.Ar(139,230.034532562),new n.Ar(140,231.035878898),new n.Ar(141,232.03858172),new n.Ar(142,233.040240235),new n.Ar(143,234.043302325),new n.Ar(144,235.045436759),new n.Ar(145,236.048675176),new n.Ar(146,237.05113943),new n.Ar(147,238.054497046),new n.Ar(148,239.05713),new n.Ar(149,240.06098)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(126,218.023487),new n.Ar(127,219.024915423),new n.Ar(128,220.024712),new n.Ar(129,221.026351),new n.Ar(130,222.02607),new n.Ar(131,223.027722956),new n.Ar(132,224.027590139),new n.Ar(133,225.029384369),new n.Ar(134,226.02933975),new n.Ar(135,227.031140069),new n.Ar(136,228.031366357),new n.Ar(137,229.033496137),new n.Ar(138,230.033927392),new n.Ar(139,231.036289158),new n.Ar(140,232.03714628),new n.Ar(141,233.039628196),new n.Ar(142,234.040945606),new n.Ar(143,235.043923062),new n.Ar(144,236.045561897),new n.Ar(145,237.048723955),new n.Ar(146,238.050782583),new n.Ar(147,239.054287777),new n.Ar(148,240.056585734),new n.Ar(149,241.06033),new n.Ar(150,242.062925)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(132,225.033899689),new n.Ar(133,226.035129),new n.Ar(134,227.034958261),new n.Ar(135,228.03618),new n.Ar(136,229.036246866),new n.Ar(137,230.037812591),new n.Ar(138,231.038233161),new n.Ar(139,232.040099),new n.Ar(140,233.04073235),new n.Ar(141,234.042888556),new n.Ar(142,235.044055876),new n.Ar(143,236.046559724),new n.Ar(144,237.048167253),new n.Ar(145,238.050940464),new n.Ar(146,239.052931399),new n.Ar(147,240.056168828),new n.Ar(148,241.058246266),new n.Ar(149,242.061635),new n.Ar(150,243.064273),new n.Ar(151,244.06785)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(134,228.038727686),new n.Ar(135,229.040138934),new n.Ar(136,230.039645603),new n.Ar(137,231.041258),new n.Ar(138,232.041179445),new n.Ar(139,233.04298757),new n.Ar(140,234.043304681),new n.Ar(141,235.0452815),new n.Ar(142,236.046048088),new n.Ar(143,237.048403774),new n.Ar(144,238.0495534),new n.Ar(145,239.052156519),new n.Ar(146,240.05380746),new n.Ar(147,241.056845291),new n.Ar(148,242.058736847),new n.Ar(149,243.061997013),new n.Ar(150,244.06419765),new n.Ar(151,245.067738657),new n.Ar(152,246.070198429),new n.Ar(153,247.07407)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(136,231.04556),new n.Ar(137,232.04659),new n.Ar(138,233.046472),new n.Ar(139,234.047794),new n.Ar(140,235.048029),new n.Ar(141,236.049569),new n.Ar(142,237.049970748),new n.Ar(143,238.051977839),new n.Ar(144,239.053018481),new n.Ar(145,240.055287826),new n.Ar(146,241.056822944),new n.Ar(147,242.059543039),new n.Ar(148,243.061372686),new n.Ar(149,244.064279429),new n.Ar(150,245.066445398),new n.Ar(151,246.069768438),new n.Ar(152,247.072086),new n.Ar(153,248.075745),new n.Ar(154,249.07848)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(137,233.0508),new n.Ar(138,234.05024),new n.Ar(139,235.051591),new n.Ar(140,236.051405),new n.Ar(141,237.052891),new n.Ar(142,238.053016298),new n.Ar(143,239.054951),new n.Ar(144,240.055519046),new n.Ar(145,241.057646736),new n.Ar(146,242.058829326),new n.Ar(147,243.061382249),new n.Ar(148,244.062746349),new n.Ar(149,245.065485586),new n.Ar(150,246.067217551),new n.Ar(151,247.070346811),new n.Ar(152,248.072342247),new n.Ar(153,249.075947062),new n.Ar(154,250.078350687),new n.Ar(155,251.082277873),new n.Ar(156,252.08487)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(138,235.05658),new n.Ar(139,236.05733),new n.Ar(140,237.057127),new n.Ar(141,238.058266),new n.Ar(142,239.058362),new n.Ar(143,240.059749),new n.Ar(144,241.060223),new n.Ar(145,242.06205),new n.Ar(146,243.06300157),new n.Ar(147,244.065167882),new n.Ar(148,245.066355386),new n.Ar(149,246.068666836),new n.Ar(150,247.070298533),new n.Ar(151,248.07308),new n.Ar(152,249.074979937),new n.Ar(153,250.078310529),new n.Ar(154,251.08075344),new n.Ar(155,252.084303),new n.Ar(156,253.08688),new n.Ar(157,254.0906)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(139,237.06207),new n.Ar(140,238.06141),new n.Ar(141,239.062579),new n.Ar(142,240.062295),new n.Ar(143,241.063716),new n.Ar(144,242.063688713),new n.Ar(145,243.065421),new n.Ar(146,244.06599039),new n.Ar(147,245.068039),new n.Ar(148,246.068798807),new n.Ar(149,247.070992043),new n.Ar(150,248.07217808),new n.Ar(151,249.074846818),new n.Ar(152,250.076399951),new n.Ar(153,251.079580056),new n.Ar(154,252.081619582),new n.Ar(155,253.085126791),new n.Ar(156,254.087316198),new n.Ar(157,255.091039),new n.Ar(158,256.09344)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(141,240.06892),new n.Ar(142,241.068662),new n.Ar(143,242.069699),new n.Ar(144,243.069631),new n.Ar(145,244.070969),new n.Ar(146,245.071317),new n.Ar(147,246.072965),new n.Ar(148,247.07365),new n.Ar(149,248.075458),new n.Ar(150,249.076405),new n.Ar(151,250.078654),new n.Ar(152,251.079983592),new n.Ar(153,252.082972247),new n.Ar(154,253.084817974),new n.Ar(155,254.088016026),new n.Ar(156,255.090266386),new n.Ar(157,256.093592),new n.Ar(158,257.095979)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(142,242.07343),new n.Ar(143,243.07451),new n.Ar(144,244.074077),new n.Ar(145,245.075375),new n.Ar(146,246.075281634),new n.Ar(147,247.076819),new n.Ar(148,248.077184411),new n.Ar(149,249.079024),new n.Ar(150,250.079514759),new n.Ar(151,251.081566467),new n.Ar(152,252.082460071),new n.Ar(153,253.085176259),new n.Ar(154,254.086847795),new n.Ar(155,255.089955466),new n.Ar(156,256.091766522),new n.Ar(157,257.095098635),new n.Ar(158,258.097069),new n.Ar(159,259.100588)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(144,245.081017),new n.Ar(145,246.081933),new n.Ar(146,247.081804),new n.Ar(147,248.082909),new n.Ar(148,249.083002),new n.Ar(149,250.084488),new n.Ar(150,251.084919),new n.Ar(151,252.08663),new n.Ar(152,253.08728),new n.Ar(153,254.089725),new n.Ar(154,255.091075196),new n.Ar(155,256.094052757),new n.Ar(156,257.095534643),new n.Ar(157,258.098425321),new n.Ar(158,259.100503),new n.Ar(159,260.103645)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(147,249.087823),new n.Ar(148,250.087493),new n.Ar(149,251.08896),new n.Ar(150,252.088965909),new n.Ar(151,253.090649),new n.Ar(152,254.090948746),new n.Ar(153,255.093232449),new n.Ar(154,256.094275879),new n.Ar(155,257.096852778),new n.Ar(156,258.0982),new n.Ar(157,259.101024),new n.Ar(158,260.102636),new n.Ar(159,261.105743),new n.Ar(160,262.10752)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(148,251.09436),new n.Ar(149,252.09533),new n.Ar(150,253.095258),new n.Ar(151,254.096587),new n.Ar(152,255.096769),new n.Ar(153,256.098763),new n.Ar(154,257.099606),new n.Ar(155,258.101883),new n.Ar(156,259.10299),new n.Ar(157,260.105572),new n.Ar(158,261.106941),new n.Ar(159,262.109692),new n.Ar(160,263.111394)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(149,253.100679),new n.Ar(150,254.100166),new n.Ar(151,255.101492),new n.Ar(152,256.101179573),new n.Ar(153,257.103072),new n.Ar(154,258.103568),new n.Ar(155,259.105628),new n.Ar(156,260.106434),new n.Ar(157,261.108752),new n.Ar(158,262.109918),new n.Ar(159,263.11254),new n.Ar(160,264.113978)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(150,255.107398),new n.Ar(151,256.10811),new n.Ar(152,257.107858),new n.Ar(153,258.109438),new n.Ar(154,259.109721),new n.Ar(155,260.111427),new n.Ar(156,261.112106),new n.Ar(157,262.114153),new n.Ar(158,263.115078),new n.Ar(159,264.117473),new n.Ar(160,265.118659)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(152,258.113151),new n.Ar(153,259.114652),new n.Ar(154,260.114435447),new n.Ar(155,261.116199),new n.Ar(156,262.116477),new n.Ar(157,263.118313),new n.Ar(158,264.118924),new n.Ar(159,265.121066),new n.Ar(160,266.121928)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(153,260.121803),new n.Ar(154,261.1218),new n.Ar(155,262.123009),new n.Ar(156,263.123146),new n.Ar(157,264.12473),new n.Ar(158,265.125198),new n.Ar(159,266.127009),new n.Ar(160,267.12774)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(155,263.12871),new n.Ar(156,264.128408258),new n.Ar(157,265.130001),new n.Ar(158,266.130042),new n.Ar(159,267.131774),new n.Ar(160,268.132156),new n.Ar(161,269.134114)]),j(v(Q.v8,1),xt,3,0,[new n.Ar(156,265.136567),new n.Ar(157,266.13794),new n.Ar(158,267.137526),new n.Ar(159,268.138816),new n.Ar(160,269.139106),new n.Ar(161,270.140723),new n.Ar(162,271.141229)])]);},n.Dr=function(t,A){n.Cr();var e,r;for(r=A-t,e=0;e<n.Br[t].length;e++)if(n.Br[t][e].b==r)return n.Br[t][e].a;return NaN;},n.Er=function(t,A){var e;if(t.b.length!=A.b.length)return t.b.length<A.b.length?-1:1;for(e=0;e<t.b.length;e++)if(t.b[e]!=A.b[e])return t.b[e]<A.b[e]?-1:1;return 0;},n.Fr=function(t){var A,e,r,P,i,D;for(e=0,P=t.a,i=0,D=P.length;i<D;++i)r=P[i],r&&++e;for(t.b=B(Q.S7,H,6,e,15,1),e=0,A=0;A<t.a.length;A++)t.a[A]&&(t.b[e++]=A);},n.Gr=function(t,A){var e;for(n.Fr(t),e=0;e<t.b.length;e++)if(A[t.b[e]])return  true;return  false;},n.Hr=function(t){this.a=B(Q.xfb,Tt,6,t,16,1);},I(239,1,{239:1,32:1},n.Hr),L.Sb=function(t){return n.Er(this,t);},Q.w8=N(239),n.Mr=function(){n.Mr=i1,n.Lr=j(v(Q.Q7,1),jt,6,15,[0,1.00794,4.0026,6.941,9.0122,10.811,12.011,14.007,15.999,18.998,20.18,22.99,24.305,26.982,28.086,30.974,32.066,35.453,39.948,39.098,40.078,44.956,47.867,50.942,51.996,54.938,55.845,58.933,58.693,63.546,65.39,69.723,72.61,74.922,78.96,79.904,83.8,85.468,87.62,88.906,91.224,92.906,95.94,98.906,101.07,102.91,106.42,107.87,112.41,114.82,118.71,121.76,127.6,126.9,131.29,132.91,137.33,138.91,140.12,140.91,144.24,146.92,150.36,151.96,157.25,158.93,162.5,164.93,167.26,168.93,173.04,174.97,178.49,180.95,183.84,186.21,190.23,192.22,195.08,196.97,200.59,204.38,207.2,208.98,209.98,209.99,222.02,223.02,226.03,227.03,232.04,231.04,238.03,237.05,239.05,241.06,244.06,249.08,252.08,252.08,257.1,258.1,259.1,262.11,267.12,268.13,271.13,270.13,277.15,276.15,nr,nr,283.17,285.18,er,er,293.2,rr,rr,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.0141,3.016,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71.0787,156.18828,114.10364,115.0877,103.1447,128.13052,129.11458,57.05182,137.14158,113.15934,113.15934,128.17428,131.19846,147.17646,97.11658,87.0777,101.10458,186.2134,163.17546,99.13246]),n.Jr=j(v(Q.Q7,1),jt,6,15,[0,1.007825,4.0026,7.016003,9.012182,11.009305,12,14.003074,15.994915,18.998403,19.992435,22.989767,23.985042,26.98153,27.976927,30.973762,31.97207,34.968852,39.962384,38.963707,39.962591,44.95591,47.947947,50.943962,51.940509,54.938047,55.934939,58.933198,57.935346,62.939598,63.929145,68.92558,73.921177,74.921594,79.91652,78.918336,83.911507,84.911794,87.905619,88.905849,89.904703,92.906377,97.905406,89.92381,101.904348,102.9055,105.903478,106.905092,113.903357,114.90388,119.9022,120.903821,129.906229,126.904473,131.904144,132.905429,137.905232,138.906346,139.905433,140.907647,141.907719,135.92398,151.919729,152.921225,157.924099,158.925342,163.929171,164.930319,165.93029,168.934212,173.938859,174.94077,179.946545,180.947992,183.950928,186.955744,191.961467,192.962917,194.964766,196.966543,201.970617,204.974401,207.976627,208.980374,193.98818,195.99573,199.9957,201.00411,206.0038,210.00923,232.038054,216.01896,238.050784,229.03623,232.041169,237.05005,238.05302,242.06194,240.06228,243.06947,243.07446,248.08275,251.08887,253.09515,257.10295,257.10777,271.13,270.13,277.15,276.15,nr,nr,283.17,285.18,er,er,291.2,rr,rr,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.014,3.01605,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]),n.Kr=j(v(Q.S7,1),H,6,15,[6,1,7,8]);},n.Nr=function(t){var A,e;for(e=t.a,A=0;A<t.b.length;A++)e+=t.b[A]*n.Jr[t.c[A]];return e;},n.Or=function(t){var A,e;for(A=new h.Ekb(),e=0;e<t.b.length;e++)h.Dkb(A,(n.Rk(),n.Lk)[t.c[e]]),t.b[e]>1&&h.Ckb(A,t.b[e]);return A.a;},n.Pr=function(t){var A,e;for(e=t.d,A=0;A<t.b.length;A++)e+=t.b[A]*n.Lr[t.c[A]];return e;},n.Qr=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(n.bw(t,1),P=B(Q.S7,H,6,191,15,1),e=0;e<t.q;e++)switch(t.C[e]){case 171:P[1]+=5,P[6]+=3,P[7]+=1,P[8]+=1;break;case 172:P[1]+=12,P[6]+=6,P[7]+=4,P[8]+=1;break;case 173:P[1]+=6,P[6]+=4,P[7]+=2,P[8]+=2;break;case 174:P[1]+=5,P[6]+=4,P[7]+=1,P[8]+=3;break;case 175:P[1]+=5,P[6]+=3,P[7]+=1,P[8]+=1,P[16]+=1;break;case 176:P[1]+=8,P[6]+=5,P[7]+=2,P[8]+=2;break;case 177:P[1]+=7,P[6]+=5,P[7]+=1,P[8]+=3;break;case 178:P[1]+=3,P[6]+=2,P[7]+=1,P[8]+=1;break;case 179:P[1]+=7,P[6]+=6,P[7]+=3,P[8]+=1;break;case 181:case 180:P[1]+=11,P[6]+=6,P[7]+=1,P[8]+=1;break;case 182:P[1]+=12,P[6]+=6,P[7]+=2,P[8]+=1;break;case 183:P[1]+=9,P[6]+=5,P[7]+=1,P[8]+=1,P[16]+=1;break;case 184:P[1]+=9,P[6]+=9,P[7]+=1,P[8]+=1;break;case 185:P[1]+=7,P[6]+=5,P[7]+=1,P[8]+=1;break;case 186:P[1]+=5,P[6]+=3,P[7]+=1,P[8]+=2;break;case 187:P[1]+=7,P[6]+=4,P[7]+=1,P[8]+=2;break;case 188:P[1]+=10,P[6]+=11,P[7]+=2,P[8]+=1;break;case 189:P[1]+=9,P[6]+=9,P[7]+=1,P[8]+=2;break;case 190:P[1]+=9,P[6]+=5,P[7]+=1,P[8]+=1;break;case 1:switch(t.A[e]){case 0:case 1:++P[1];break;case 2:++P[151];break;case 3:++P[152];}break;default:++P[t.C[e]];}for(r=0;r<t.q;r++)t.C[r]>=171&&t.C[r]<=190?P[1]+=2-n.vp(t,r):P[1]+=n.op(t,r);for(u=0,l=1;l<=190;l++)P[l]!=0&&++u;for(this.b=B(Q.S7,H,6,u,15,1),this.c=B(Q.S7,H,6,u,15,1),u=0,o=0;o<n.Kr.length;o++)P[n.Kr[o]]!=0&&(this.b[u]=P[n.Kr[o]],this.c[u]=n.Kr[o],++u,P[n.Kr[o]]=0);for(;;){for(y="zzz",U=-1,D=1;D<=190;D++)P[D]>0&&h.Yjb(y,(n.Rk(),n.Lk)[D])>0&&(y=(n.Rk(),n.Lk)[D],U=D);if(U==-1)break;this.b[u]=P[U],this.c[u]=U,++u,P[U]=0;}for(this.a=0,this.d=0,A=0;A<t.f;A++)t.C[A]!=1&&t.A[A]!=0&&(D=t.C[A],i=t.A[A],this.a+=n.Dr(D,i)-n.Jr[D],this.d+=n.Dr(D,i)-n.Lr[D]);},I(153,1,{153:1}),L.mb=function(t){var A;if(t===this)return  true;if(!Zt(t,153))return  false;for(A=0;A<this.b.length;A++)if(this.b[A]!=t.b[A])return  false;return  true;},L.a=0,L.d=0,Q.x8=N(153),n.Sr=function(t){switch(t){case 6:return 1;case 53:return 2;case 33:return 3;case 34:return 4;case 35:return 5;case 15:return 6;case 16:return 7;case 17:return 8;case 7:return 9;case 8:return 10;case 9:return 11;}return 0;},n.Tr=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V;for(n.bw(t,7),J=0,c=B(Q.xfb,Tt,6,t.f,16,1),V=B(Q.S7,H,6,t.f,15,1),i=0;i<t.g;i++)e=t.D[0][i],r=t.D[1][i],D=t.s[e],u=t.s[r],D!=0&&u!=0&&D<0^u<0&&(c[e]=true,c[r]=true),V[e]+=n.fm(t,i),V[r]+=n.fm(t,i);for(o=0;o<t.f;o++)if(J+=t.s[o],t.s[o]==1){if(t.C[o]==7&&!c[o]){if(V[o]<=3)J-=1,t.s[o]=0,t.T=0,t.j[o]!=t.e[o]&&(n.ql(t,n.Zo(t,o,t.e[o]-1)),n.bw(t,7));else if(o<t.f&&n.ot(t.p,o)){for(F=B(Q.xfb,Tt,6,t.f,16,1),S=B(Q.xfb,Tt,6,t.g,16,1),n.Io(t,o,true,F,S),U=0;U<t.f;U++)if(F[U]&&t.C[U]==7&&t.s[U]==0&&V[U]==2&&n.Wr(t,S,o,U)){J-=1;break;}}}}else if(t.s[o]<0&&(t.C[o]==6||t.C[o]==7||t.C[o]==8||t.C[o]==16))if(c[o]){for(O=B(Q.S7,H,6,16,15,1),O[0]=o,G=1,x=0;x<G;){for(P=0;P<t.r;P++)if(A=-1,t.D[0][P]==O[x]?A=t.D[1][P]:t.D[1][P]==O[x]&&(A=t.D[0][P]),A!=-1&&t.s[A]!=0){for(y=false,g=0;g<G;g++)if(A==O[g]){y=true;break;}y||(G==O.length&&(l=B(Q.S7,H,6,2*O.length,15,1),h.Skb(O,0,l,0,O.length),O=l),O[G++]=A);}++x;}for(T=0,k=0;k<G;k++)T+=n.Bl(t,O[k]);if(T<0){for(R=-1,E=99,b=0;b<G;b++)n.Bl(t,O[b])<0&&E>n.Sr(n.Tl(t,O[b]))&&(E=n.Sr(n.Tl(t,O[b])),R=O[b]);R!=-1&&(J-=t.s[R],t.s[R]=0,t.T=0);}}else J-=t.s[o],t.s[o]=0,t.T=0;return J>0&&(J=n.Ur(t,J)),J;},n.Ur=function(t,A){var e,r,P,i;if(A>0&&(A=n.Xr(t,A,9)),A>0&&(A=n.Xr(t,A,17)),A>0&&(A=n.Xr(t,A,35)),A>0&&(A=n.Xr(t,A,53)),A>0){for(e=0;e<t.f;e++)if(t.s[e]>0){for(P=false,i=0;i<t.j[e];i++)if(r=t.i[e][i],t.s[r]==0&&n.Hm(t,r)&&n.op(t,r)>0){if(t.s[r]=-1,t.T=0,--A,A==0)return 0;P=true;break;}if(P)continue;}}return A>0&&(A=n.Vr(t,A,8,16)),A>0&&(A=n.Vr(t,A,8,15)),A>0&&(A=n.Vr(t,A,8,6)),A>0&&(A=n.Vr(t,A,7,16)),A;},n.Vr=function(t,A,e,r){var P,i,D,u,o,l,U,y;for(P=0;P<t.f;P++)if(t.C[P]==e&&t.s[P]==0&&n.op(t,P)>0)for(u=false,l=0;l<t.j[P];l++){if(t.n[P][l]==1&&(i=t.i[P][l],t.C[i]==r)){for(y=false,o=false,U=0;U<t.j[i];U++){if(D=t.i[i][U],t.s[D]<0){o=true;break;}D!=P&&t.C[D]==8&&t.n[i][U]==2&&(y=true);}if(!o&&y){if(t.s[P]=-1,t.T=0,--A,A==0)return 0;u=true;}}if(u)break;}return A;},n.Wr=function(t,A,e,r){var P,i,D,u,o,l,U,y;for(y=B(Q.S7,H,6,t.g,15,1),o=B(Q.xfb,Tt,6,t.g,16,1),D=0;D<t.g;D++)y[D]=t.H[D],A[D]&&(o[D]=true,t.H[D]=1,t.T=0);for(P=0;P<t.f;P++)if(P!=r&&t.o[P]==0&&o[t.k[r][0]])for(U=0;U<t.j[P];U++)u=t.k[P][U],t.H[u]=1,t.T=0,o[u]=false;for(l=0;l<t.j[e];l++)u=t.k[e][l],t.H[u]=1,t.T=0,o[u]=false;if(n.Ge(new n.Ve(t),o,false))return t.s[e]=0,t.T=0,t.j[r]!=t.e[r]&&n.ql(t,n.Zo(t,r,t.e[r]-1)),n.bw(t,7),true;for(i=0;i<t.g;i++)n.Mn(t,i,y[i]);return  false;},n.Xr=function(t,A,e){var r;for(r=0;r<t.f;r++)if(t.C[r]==e&&t.s[r]==0&&t.j[r]==0&&(t.s[r]=-1,t.T=0,--A,A==0))return 0;return A;},n.Yr=function(t,A){var e,r;for(r=Q0.Fnb(t.b,A),e=w.UAb(r).length;e<10;e++)h.Gkb(t.a,32);h.Kkb(t.a,r);},n.Zr=function(t,A){var e,r,P;if(A<0||A>999){h.Kkb(t.a,"  ?");return;}for(e=false,r=0;r<3;r++)P=A/100|0,P==0?r==2||e?h.Gkb(t.a,48):h.Gkb(t.a,32):(h.Gkb(t.a,48+P&n1),e=true),A=10*(A%100);},n.$r=function(t,A){Ct.uhb(A,t.a.a);},n._r=function(t){n.as.call(this,t);},n.as=function(t){n.cs.call(this,t,new h.Nkb());},n.bs=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt,Gt,dt,Bt,b1,f1,St,gt,g1,q1,U1;for(this.b=new Q0.Knb((ct=Pr,ct)),Gt=(h.Rkb(),m1),n.bw(t,15),bt=true,P=0;P<t.f;P++)if((t.u[P]&3)!=0&&(t.u[P]&3)!=3&&(t.u[P]&l1)>>18!=1){bt=false;break;}if(Et=-1,bt){for(K=B(Q.S7,H,6,32,15,1),Ht=0,i=0;i<t.f;i++)if((t.u[i]&3)!=0&&(t.u[i]&3)!=3&&(t.u[i]&l1)>>18==1){lt=(t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&v0)>>20,++K[lt],Ht<K[lt]&&(Ht=K[lt],Et=lt);break;}}for(this.a=e||new h.Mkb(),Nt=t.P!=null?t.P:"",h.Kkb(this.a,Nt+Gt),h.Kkb(this.a,"Actelion Java MolfileCreator 1.0"+Gt+Gt),n.Zr(this,t.q),n.Zr(this,t.r),h.Kkb(this.a,"  0  0"),n.Zr(this,bt?0:1),h.Kkb(this.a,"  0  0  0  0  0999 V2000"+Gt),ut=t.q==1,l=1;l<t.q;l++)if(t.J[l].a!=t.J[0].a||t.J[l].b!=t.J[0].b||t.J[l].c!=t.J[0].c){ut=true;break;}if(it=1,ut)if(A!=0)it=A;else if(R=n.Wl(t,t.q,t.r,(n.Rk(),n.Qk)),R!=0)(R<1||R>3)&&(it=1.5/R);else {for(Jt=L0,i=1;i<t.q;i++)for(o=0;o<i;o++)V=t.J[o].a-t.J[i].a,At=t.J[o].b-t.J[i].b,ot=t.J[o].c-t.J[i].c,J=V*V+At*At+ot*ot,Jt>J&&(Jt=J);it=3/Jt;}for(U=0;U<t.q;U++)ut?(n.Yr(this,it*t.J[U].a),n.Yr(this,it*-t.J[U].b),n.Yr(this,it*-t.J[U].c)):h.Kkb(this.a,"    0.0000    0.0000    0.0000"),(t.v==null?null:t.v[U])!=null?h.Kkb(this.a," L  "):Rt(z(t.B[U],1),0)?h.Kkb(this.a," A  "):t.C[U]>=129&&t.C[U]<=144||t.C[U]==154?h.Kkb(this.a," R# "):(k=(n.Rk(),n.Lk)[t.C[U]],h.Kkb(this.a," "+k),w.UAb(k).length==1?h.Kkb(this.a,"  "):w.UAb(k).length==2&&h.Kkb(this.a," ")),h.Kkb(this.a," 0  0  0"),Dt=z(X0,t.B[U]),nt(Dt,0)==0?h.Kkb(this.a,"  0"):nt(Dt,384)==0?h.Kkb(this.a,"  3"):nt(Dt,128)==0?h.Kkb(this.a,"  2"):nt(Dt,1792)==0?h.Kkb(this.a,"  1"):nt(Dt,1664)==0&&h.Kkb(this.a,"  2"),h.Kkb(this.a,Rt(z(t.B[U],U0),0)?"  1":"  0"),U1=((t.u[U]&bA)>>>27)-1,U1==-1?h.Kkb(this.a,"  0"):U1==0?h.Kkb(this.a," 15"):n.Zr(this,U1),h.Kkb(this.a,"  0  0  0"),n.Zr(this,d.Math.abs(t.w[U])),h.Kkb(this.a,"  0  0"+Gt);for(E=0;E<t.r;E++){switch(t.H[E]){case 1:Bt=1,gt=0;break;case 2:Bt=2,gt=0;break;case 4:Bt=3,gt=0;break;case 129:Bt=1,gt=6;break;case 257:Bt=1,gt=1;break;case 386:Bt=2,gt=3;break;case 8:Bt=4,gt=0;break;case 16:Bt=8,gt=0;break;default:Bt=1,gt=0;}bt&&(gt==1||gt==6)&&(i=t.D[0][E],(t.u[i]&l1)>>18==2?gt=0:(t.u[i]&l1)>>18==1&&((t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&v0)>>20)!=Et&&(gt=4)),O=t.G[E]&31,O!=0&&(O==8?Bt=4:O==3?Bt=5:O==9?Bt=6:O==10?Bt=7:Bt=8),f1=t.G[E]&384,q1=f1==0?0:f1==256?1:2,n.Zr(this,1+t.D[0][E]),n.Zr(this,1+t.D[1][E]),n.Zr(this,Bt),n.Zr(this,gt),h.Kkb(this.a,"  0"),n.Zr(this,q1),h.Kkb(this.a,"  0"+Gt);}for(dt=0,y=0;y<t.q;y++)t.s[y]!=0&&++dt;if(dt!=0)for(G=0,i=0;i<t.q;i++)t.s[i]!=0&&(G==0&&(h.Kkb(this.a,"M  CHG"),n.Zr(this,d.Math.min(8,dt))),h.Kkb(this.a," "),n.Zr(this,i+1),x=t.s[i],x<0?(h.Kkb(this.a,"  -"),x=-x):h.Kkb(this.a,"   "),h.Gkb(this.a,48+x&n1),--dt,(++G==8||dt==0)&&(G=0,h.Kkb(this.a,Gt)));for(dt=0,T=0;T<t.q;T++)t.A[T]==0||++dt;if(dt!=0)for(G=0,i=0;i<t.q;i++)t.A[i]!=0&&(G==0&&(h.Kkb(this.a,"M  ISO"),n.Zr(this,d.Math.min(8,dt))),h.Kkb(this.a," "),n.Zr(this,i+1),h.Kkb(this.a," "),n.Zr(this,t.A[i]),--dt,(++G==8||dt==0)&&(G=0,h.Kkb(this.a,Gt)));for(dt=0,c=0;c<t.q;c++)(t.u[c]&48)!=0&&++dt;if(dt!=0){for(G=0,i=0;i<t.q;i++)if((t.u[i]&48)!=0){switch(G==0&&(h.Kkb(this.a,"M  RAD"),n.Zr(this,d.Math.min(8,dt))),h.Kkb(this.a," "),n.Zr(this,i+1),t.u[i]&48){case 16:h.Kkb(this.a,"   1");break;case 32:h.Kkb(this.a,"   2");break;case 48:h.Kkb(this.a,"   3");}--dt,(++G==8||dt==0)&&(G=0,h.Kkb(this.a,Gt));}}for(dt=0,b=0;b<t.q;b++)(t.C[b]>=129&&t.C[b]<=144||t.C[b]==154)&&++dt;if(dt!=0)for(G=0,i=0;i<t.q;i++)S=t.C[i],(S>=129&&S<=144||S==154)&&(G==0&&(h.Kkb(this.a,"M  RGP"),n.Zr(this,d.Math.min(8,dt))),h.Kkb(this.a," "),n.Zr(this,i+1),h.Kkb(this.a," "),n.Zr(this,S==154?0:S>=142?S-141:S-125),--dt,(++G==8||dt==0)&&(G=0,h.Kkb(this.a,Gt)));if(t.K){for(dt=0,i=0;i<t.q;i++)Rt(z(t.B[i],120),0)&&++dt;if(dt!=0)for(G=0,o=0;o<t.q;o++)b1=z(t.B[o],120),nt(b1,0)!=0&&(G==0&&(h.Kkb(this.a,"M  RBC"),n.Zr(this,d.Math.min(8,dt))),h.Kkb(this.a," "),n.Zr(this,o+1),nt(b1,112)==0?h.Kkb(this.a,"  -1"):nt(b1,8)==0?h.Kkb(this.a,"   1"):nt(b1,104)==0?h.Kkb(this.a,"   2"):nt(b1,88)==0?h.Kkb(this.a,"   3"):nt(b1,56)==0&&h.Kkb(this.a,"   4"),--dt,(++G==8||dt==0)&&(G=0,h.Kkb(this.a,Gt)));for(g=0;g<t.q;g++)if(F=t.v==null?null:t.v[g],F!=null){for(h.Kkb(this.a,"M  ALS "),n.Zr(this,g+1),n.Zr(this,F.length),h.Kkb(this.a,Rt(z(t.B[g],1),0)?" T ":" F "),Qt=0;Qt<F.length;Qt++)switch(st=(n.Rk(),n.Lk)[F[Qt]],w.UAb(st).length){case 1:h.Kkb(this.a,st+"   ");break;case 2:h.Kkb(this.a,st+"  ");break;case 3:h.Kkb(this.a,st+" ");break;default:h.Kkb(this.a,"   ?");}h.Kkb(this.a,Gt);}for(dt=0,D=0;D<t.q;D++)Rt(z(t.B[D],6144),0)&&++dt;if(dt!=0)for(G=0,o=0;o<t.q;o++)g1=z(t.B[o],6144),nt(g1,0)!=0&&(G==0&&(h.Kkb(this.a,"M  SUB"),n.Zr(this,d.Math.min(8,dt))),h.Kkb(this.a," "),n.Zr(this,o+1),Rt(z(g1,Z1),0)?h.Kkb(this.a,"   "+(t.e[o]+1)):h.Kkb(this.a,"  -2"),--dt,(++G==8||dt==0)&&(G=0,h.Kkb(this.a,Gt)));}for(dt=0,u=0;u<t.q;u++)(t.t==null||t.t[u]==null?null:h.akb((Vt=t.t[u],O1.Enb(),Vt)))!=null&&++dt;if(dt!=0)for(St=0,r=0;r<t.q;r++)st=t.t==null||t.t[r]==null?null:h.akb((wt=t.t[r],O1.Enb(),wt)),st!=null&&(++St,h.Kkb(this.a,"M  STY  1 "),n.Zr(this,St),h.Kkb(this.a," DAT"),h.Kkb(this.a,Gt),h.Kkb(this.a,"M  SLB  1 "),n.Zr(this,St),h.Kkb(this.a," "),n.Zr(this,St),h.Kkb(this.a,Gt),h.Kkb(this.a,"M  SAL "),n.Zr(this,St),h.Kkb(this.a,"  1 "),n.Zr(this,r+1),h.Kkb(this.a,Gt),h.Kkb(this.a,"M  SDT "),n.Zr(this,St),h.Kkb(this.a," "),h.Kkb(this.a,ir),h.Kkb(this.a,Gt),h.Kkb(this.a,"M  SDD "),n.Zr(this,St),h.Kkb(this.a," "),n.Yr(this,t.J[r].a),n.Yr(this,t.J[r].b),h.Kkb(this.a,"    DA    ALL  1       5"),h.Kkb(this.a,Gt),h.Kkb(this.a,"M  SED "),n.Zr(this,St),h.Kkb(this.a," "),h.Kkb(this.a,st),h.Kkb(this.a,Gt));h.Kkb(this.a,vA+Gt);},n.cs=function(t,A){n.bs.call(this,t,0,A);},I(112,1,{},n._r,n.bs),Q.z8=N(112),n.ds=function(t,A,e,r,P,i){var D,u,o,l;switch(l=1,u=false,P){case 1:l=257;break;case 3:l=386;break;case 4:l=257,u=true;break;case 6:l=129;break;default:switch(r){case 1:l=1;break;case 2:l=2;break;case 3:l=4;break;case 4:l=8;break;case 9:l=16;}}if(D=n.Wk(t.j,A,e,l),o=0,u&&n.on(t.j,A,1,-1),r>4)switch(r){case 5:o|=3;break;case 6:o|=9;break;case 7:o|=10;break;case 8:l!=16&&(o|=127);}return i==1&&(o|=256),i==2&&(o|=128),o!=0&&n.Ln(t.j,D,o,true),D;},n.es=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(D=B(Q.S7,H,6,t.j.q,15,1),e=0;e<t.j.q;e++)D[e]=-n.Bl(t.j,e);for(n.Ge(new n.Ve(t.j),null,true),r=0;r<t.j.q;r++)D[r]+=n.Bl(t.j,r);for(A=0;A<t.j.q;A++)if(D[A]!=0)for(u=-D[A],P=0;P<t.j.r;P++)for(l=0;l<2;l++)u>0&&n.im(t.j,P)==16&&n.Zl(t.j,1-l,P)==A&&(y=n.Zl(t.j,l,P),n.Lm(t.j,y)&&(U=n.gs(t,y),i=n.Bl(t.j,y),i<U&&(o=d.Math.min(u,U-i),n.jn(t.j,y,i+o),u-=o)));},n.fs=function(t,A){return t.j=null,n.As(t,new Ct.ghb(new Ct.rhb(A)))?t.j:null;},n.gs=function(t,A){var e,r;return e=n.Tl(t.j,A),r=e<(n.Rk(),n.Nk).length?n.Nk[e]:null,r==null?0:r[r.length-1];},n.hs=function(t,A){var e;return e=t.b?s.gqb(t.b,h.sjb(A)):null,e?e.a:A-1;},n.is=function(t,A){var e;return e=t.c?s.gqb(t.c,h.sjb(A)):null,e?e.a:A-1;},n.js=function(t,A){var e,r;if(A!=null)for(n.bw(t.j,1),e=0;e<t.j.f;e++)A[e]!=0&&(r=n.mm(t.j,e,n.vp(t.j,e)),A[e]==15?r>=0&&n.gn(t.j,e,0):A[e]!=n.om(t.j,e)&&n.gn(t.j,e,A[e]-r));},n.ks=function(t,A){var e,r;for(e=A;e<w.UAb(t).length;e++){if(r=(w.PAb(e,w.UAb(t).length),w.UAb(t).charCodeAt(e)),r==32||r==9)return  -1;if(r==61)return e;}return  -1;},n.ls=function(t,A){var e;if(A==-1)return  -1;for(e=A+1;e<w.UAb(t).length;e++)if(w.PAb(e,w.UAb(t).length),w.UAb(t).charCodeAt(e)!=32&&(w.PAb(e,w.UAb(t).length),w.UAb(t).charCodeAt(e)!=9))return e;return  -1;},n.ms=function(t,A){var e,r,P,i;for(e=false,P=false,r=A;r<w.UAb(t).length;r++)if(i=(w.PAb(r,w.UAb(t).length),w.UAb(t).charCodeAt(r)),i==40?e=true:i==41?e=false:i==34&&(P=!P),!e&&!P&&(i==32||i==9))return r;return  -1;},n.ns=function(t){return w.UAb(t).indexOf("ATOMS=(")!=-1?On:w.UAb(t).indexOf("BONDS=(")!=-1?"BONDS":null;},n.os=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot;for(g=0,k=(u=n.ms(A,g),u==-1?w.UAb(A).length:u),r=h.Oib((w.OAb(g,k,w.UAb(A).length),w.UAb(A).substr(g,k-g)),e1,Ot),g=n.ls(A,k),k=(o=n.ms(A,g),o==-1?w.UAb(A).length:o),S=(w.OAb(g,k,w.UAb(A).length),w.UAb(A).substr(g,k-g)),G=null,P=false,F=n.ws(A),F!=0&&(G=n.ps(A),F<0&&(P=true),k=d.Math.abs(F)),g=n.ls(A,k),k=(l=n.ms(A,g),l==-1?w.UAb(A).length:l),V=I0((w.OAb(g,k,w.UAb(A).length),w.UAb(A).substr(g,k-g))),g=n.ls(A,k),k=(U=n.ms(A,g),U==-1?w.UAb(A).length:U),At=I0((w.OAb(g,k,w.UAb(A).length),w.UAb(A).substr(g,k-g))),g=n.ls(A,k),k=(y=n.ms(A,g),y==-1?w.UAb(A).length:y),ot=I0((w.OAb(g,k,w.UAb(A).length),w.UAb(A).substr(g,k-g))),g=n.ls(A,k),k=(T=n.ms(A,g),T==-1?w.UAb(A).length:T),E=h.Oib((w.OAb(g,k,w.UAb(A).length),w.UAb(A).substr(g,k-g)),e1,Ot),e=n.Tk(t.j,V,-At,-ot),e+1!=r&&(!t.b&&(t.b=new s.Zwb()),s.Rwb(t.b,h.sjb(r),h.sjb(e))),G!=null&&(n.Cn(t.j,e,6),n.qn(t.j,e,G,P)),E!=0&&n.rn(t.j,e,E,false),h.ckb(S,"A")||h.ckb(S,"*")?n.vn(t.j,e,1,true):h.ckb(S,"Q")?(R=B(Q.S7,H,6,1,15,1),R[0]=6,n.qn(t.j,e,R,true)):n.Cn(t.j,e,n.po(S,67));(g=n.ls(A,k))!=-1;)if(k=(D=n.ms(A,g),D==-1?w.UAb(A).length:D),O=(w.OAb(g,k,w.UAb(A).length),w.UAb(A).substr(g,k-g)),b=h.jkb(O,X1(61)),c=(w.OAb(0,b,w.UAb(O).length),w.UAb(O).substr(0,b)),J=h.Oib((w.PAb(b+1,w.UAb(O).length+1),w.UAb(O).substr(b+1)),e1,Ot),h.ckb(c,"CHG"))n.jn(t.j,e,J);else if(h.ckb(c,"RAD"))switch(J){case 1:n.wn(t.j,e,16);break;case 2:n.wn(t.j,e,32);break;case 3:n.wn(t.j,e,48);}else if(!h.ckb(c,"CFG")){if(h.ckb(c,"MASS"))n.tn(t.j,e,J);else if(h.ckb(c,"VAL"))n.gn(t.j,e,J==-1?0:J==0?-1:J);else if(h.ckb(c,"HCOUNT"))switch(J){case 0:break;case  -1:n.vn(t.j,e,1792,true);break;case 1:n.vn(t.j,e,128,true);break;case 2:n.vn(t.j,e,384,true);break;default:n.vn(t.j,e,896,true);}else if(h.ckb(c,"SUBST")){if(J==-1)n.vn(t.j,e,B1,true);else if(J>0){for(x=0,i=0;i<t.j.r;i++)(n.Zl(t.j,0,i)==e||n.Zl(t.j,1,i)==e)&&++x;J>x&&n.vn(t.j,e,Z1,true);}}else if(h.ckb(c,"RBCNT"))switch(J){case 3:case  -1:n.vn(t.j,e,112,true);break;case 1:n.vn(t.j,e,8,true);break;case 2:n.vn(t.j,e,104,true);break;case 4:n.vn(t.j,e,56,true);}}},n.ps=function(t){var A,e,r,P,i,D,u,o;if(u=null,e=w.UAb(t).indexOf("["),r=w.UAb(t).indexOf("]",e),e>=0&&r>0){for(A=B(Q.S7,H,6,16,15,1),o=(w.OAb(e+1,r,w.UAb(t).length),w.UAb(t).substr(e+1,r-(e+1))),P=0,D=true;D&&P<16;)e=w.UAb(o).indexOf(","),i=null,e==-1?(i=o,D=false):(i=(w.OAb(0,e,w.UAb(o).length),w.UAb(o).substr(0,e)),o=(w.PAb(e+1,w.UAb(o).length+1),w.UAb(o).substr(e+1))),A[P++]=n.po(i,1);u=B(Q.S7,H,6,P,15,1),h.Skb(A,0,u,0,P);}return u;},n.qs=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R;for(b=0,g=(o=n.ms(A,b),o==-1?w.UAb(A).length:o),i=h.Oib((w.OAb(b,g,w.UAb(A).length),w.UAb(A).substr(b,g-b)),e1,Ot),b=n.ls(A,g),g=(l=n.ms(A,b),l==-1?w.UAb(A).length:l),D=h.Oib((w.OAb(b,g,w.UAb(A).length),w.UAb(A).substr(b,g-b)),e1,Ot),b=n.ls(A,g),g=(U=n.ms(A,b),U==-1?w.UAb(A).length:U),e=n.hs(t,h.Oib((w.OAb(b,g,w.UAb(A).length),w.UAb(A).substr(b,g-b)),e1,Ot)),b=n.ls(A,g),g=(y=n.ms(A,b),y==-1?w.UAb(A).length:y),r=n.hs(t,h.Oib((w.OAb(b,g,w.UAb(A).length),w.UAb(A).substr(b,g-b)),e1,Ot)),F=0,S=0;(b=n.ls(A,g))!=-1;)if(g=(u=n.ms(A,b),u==-1?w.UAb(A).length:u),k=(w.OAb(b,g,w.UAb(A).length),w.UAb(A).substr(b,g-b)),c=h.jkb(k,X1(61)),T=(w.OAb(0,c,w.UAb(k).length),w.UAb(k).substr(0,c)),R=h.Oib((w.PAb(c+1,w.UAb(k).length+1),w.UAb(k).substr(c+1)),e1,Ot),h.ckb(T,"CFG"))switch(R){case 1:F=1;break;case 2:F=D==2?3:4;break;case 3:F=6;}else h.ckb(T,"TOPO")&&(S=R);P=n.ds(t,e,r,D,F,S),P+1!=i&&(!t.c&&(t.c=new s.Zwb()),s.Rwb(t.c,h.sjb(i),h.sjb(P)));},n.rs=function(t,A){var e,r,P,i,D,u;if(u=n.ns(A),u!=null){if(D=n.ts(A,u),h.ckb(w.UAb(A).substr(0,13),"MDLV30/STEABS")){if(h.ckb(u,On))for(i=0;i<D.length;i++)n.on(t.j,n.hs(t,D[i]),0,-1);else for(P=0;P<D.length;P++)n.Gn(t.j,n.is(t,D[P]),0,-1);}else if(h.ckb(w.UAb(A).substr(0,13),"MDLV30/STERAC")){if(r=h.Oib(h.tkb(A,13,n.ms(A,13)),e1,Ot),h.ckb(u,On))for(i=0;i<D.length;i++)n.on(t.j,n.hs(t,D[i]),1,r-1);else for(P=0;P<D.length;P++)n.Gn(t.j,n.is(t,D[P]),1,r-1);}else if(h.ckb(w.UAb(A).substr(0,13),"MDLV30/STEREL")){if(r=h.Oib(h.tkb(A,13,n.ms(A,13)),e1,Ot),h.ckb(u,On))for(i=0;i<D.length;i++)n.on(t.j,n.hs(t,D[i]),2,r-1);else for(P=0;P<D.length;P++)n.Gn(t.j,n.is(t,D[P]),2,r-1);}else if(h.ckb(w.UAb(A).substr(0,13),"MDLV30/HILITE"))if(h.ckb(u,On))for(P=0;P<D.length;P++)n.kn(t.j,n.hs(t,D[P]),448);else for(P=0;P<D.length;P++)e=n.is(t,D[P]),n.kn(t.j,n.Zl(t.j,0,e),448),n.kn(t.j,n.Zl(t.j,1,e),448);}},n.ss=function(t,A){var e,r,P;t.j||h.ckb(w.UAb(A).substr(0,6),"COUNTS")&&(e=n.ls(A,n.ms(A,7)),r=h.Oib(h.tkb(A,7,n.ms(A,7)),e1,Ot),P=h.Oib(h.tkb(A,e,n.ms(A,e)),e1,Ot),t.j=new n.uw(r,P));},n.ts=function(t,A){var e,r,P,i,D,u;for(i=w.UAb(t).indexOf(A+"=(")+w.UAb(A).length+2,D=h.kkb(t,X1(41),i),P=n.ms(t,i),e=h.Oib((w.OAb(i,P,w.UAb(t).length),w.UAb(t).substr(i,P-i)),e1,Ot),u=B(Q.S7,H,6,e,15,1),r=0;r<e;r++)i=n.ls(t,P),P=n.ms(t,i),(P==-1||P>D)&&(P=D),u[r]=h.Oib((w.OAb(i,P,w.UAb(t).length),w.UAb(t).substr(i,P-i)),e1,Ot);return u;},n.us=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R;if(T=0,c=(i=n.ms(A,T),i==-1?w.UAb(A).length:i),R=(w.OAb(T,c,w.UAb(A).length),w.UAb(A).substr(T,c-T)),h.ckb("DEFAULT",R)&&(T=n.ls(A,c),c=(D=n.ms(A,T),D==-1?w.UAb(A).length:D),R=(w.OAb(T,c,w.UAb(A).length),w.UAb(A).substr(T,c-T)),h.ckb(w.UAb(R).substr(0,5),"CLASS")&&(T=n.ls(A,c),c=(u=n.ms(A,T),u==-1?w.UAb(A).length:u),R=(w.OAb(T,c,w.UAb(A).length),w.UAb(A).substr(T,c-T)),h.ckb("-",R)&&(T=n.ls(A,c),c=(o=n.ms(A,T),o==-1?w.UAb(A).length:o),R=(w.OAb(T,c,w.UAb(A).length),w.UAb(A).substr(T,c-T))))),h.Oib(R,e1,Ot),T=n.ls(A,c),c=(l=n.ms(A,T),l==-1?w.UAb(A).length:l),S=(w.OAb(T,c,w.UAb(A).length),w.UAb(A).substr(T,c-T)),!!h.ckb("DAT",S)){for(T=n.ls(A,c),c=(P=n.ms(A,T),P==-1?w.UAb(A).length:P),h.Oib((w.OAb(T,c,w.UAb(A).length),w.UAb(A).substr(T,c-T)),e1,Ot),F=new s.Zwb(),T=n.ls(A,c);T!=-1;)T=n.zs(F,A,T);r=(g=On,s.Znb(s.Mwb(F,g))),y=(k="FIELDNAME",s.Znb(s.Mwb(F,k))),U=(b="FIELDDATA",s.Znb(s.Mwb(F,b))),!(!h.ckb(ir,y)||r==null||U==null)&&(e=h.qkb(r," "),e.length==2&&n.mn(t.j,h.Oib(e[1],e1,Ot)-1,U));}},n.vs=function(t){var A,e,r,P,i,D,u,o,l;for(o=B(Q.S7,H,6,t.j.q,15,1),r=0;r<t.j.r;r++)if(n.im(t.j,r)==8)for(D=0;D<2;D++)o[n.Zl(t.j,D,r)]=1;for(P=0;P<t.j.r;P++)for(l=n.fm(t.j,P),i=0;i<2;i++)o[n.Zl(t.j,i,P)]+=l;for(e=0;e<t.j.r;e++)if(n.fm(t.j,e)==1){for(i=0;i<2;i++)if(u=n.Zl(t.j,1-i,e),n.Lm(t.j,u)&&(A=n.Zl(t.j,i,e),n.Hm(t.j,A)&&o[A]>n.om(t.j,A))){n.Mn(t.j,e,16);continue;}}},n.ws=function(t){var A,e;if(w.UAb(t).indexOf("[")>=0){if(A=w.UAb(t).indexOf(" NOT["),e=w.UAb(t).indexOf("]",A),A>=0&&e>0)return -(e+1);if(A=w.UAb(t).indexOf(" ["),e=w.UAb(t).indexOf("]",A),A>=0&&e>0)return e+1;if(A=w.UAb(t).indexOf(" 'NOT["),e=w.UAb(t).indexOf("]'",A),A>=0&&e>0)return -(e+2);if(A=w.UAb(t).indexOf(" '["),e=w.UAb(t).indexOf("]'",A),A>=0&&e>0)return e+2;h.Rkb();}return 0;},n.xs=function(t,A,e){return t.j=A,n.As(t,e);},n.ys=function(t){return w.UAb(t).length==0?0:h.Oib(t,e1,Ot);},n.zs=function(t,A,e){var r,P,i,D,u,o,l;return i=n.ks(A,e),i==-1?-1:(P=(r=n.ms(A,i+1),r==-1?w.UAb(A).length:r),D=(w.OAb(e,i,w.UAb(A).length),w.UAb(A).substr(e,i-e)),l=(w.OAb(i+1,P,w.UAb(A).length),w.UAb(A).substr(i+1,P-(i+1))),(h.ckb(w.UAb(l).substr(0,1),"(")&&(o=w.UAb(")").length,h.ckb(w.UAb(l).substr(w.UAb(l).length-o,o),")"))||h.ckb(w.UAb(l).substr(0,1),'"')&&(u=w.UAb('"').length,h.ckb(w.UAb(l).substr(w.UAb(l).length-u,u),'"')))&&(l=h.tkb(l,1,w.UAb(l).length-1)),s.Rwb(t,D,l),n.ls(A,P));},n.As=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt,Gt,dt,Bt,b1,f1,St,gt,g1,q1;f1=null,b=null;try{if(t.f=null,t.j&&(n._v(t.j),n.On(t.j,!1)),lt=V=Ct.fhb(A),lt==null||(V=Ct.fhb(A))==null||(V=Ct.fhb(A))==null||(t.k=w.UAb(V).indexOf("From CSD data. Using bond type 'Any'")!=-1,t.e=w.UAb(V).indexOf("From CSD data.")!=-1,(V=Ct.fhb(A))==null))return !1;t.g=!1,t.d=t.a;try{ut=h.Oib(h.ukb((w.OAb(0,3,w.UAb(V).length),w.UAb(V).substr(0,3))),e1,Ot),Dt=h.Oib(h.ukb((w.OAb(3,6,w.UAb(V).length),w.UAb(V).substr(3,3))),e1,Ot),Qt=n.ys(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3)))),t.d=t.d|n.ys(h.ukb((w.OAb(12,15,w.UAb(V).length),w.UAb(V).substr(12,3))))==1,t.g=w.UAb(V).length>=39&&h.ckb(w.UAb(V).substr(34,w.UAb("V3000").length),"V3000");}catch(U1){if(U1=M1(U1),Zt(U1,19))return !1;throw Lt(U1);}if(t.g)return ct=n.Bs(t,A),n.Un(t.j,lt),ct;if(!t.j&&(t.j=new n.uw(ut,Dt)),n.Un(t.j,lt),t.d||(t.j.L=!0),ut==0){for(;V!=null&&!(h.ckb(V,vA)||h.ckb(V,le)||h.ckb((w.PAb(1,w.UAb(V).length+1),w.UAb(V).substr(1)),"$"));)V=Ct.fhb(A);return !0;}for(R=0;R<ut;R++){if((V=Ct.fhb(A))==null)return !1;switch(gt=I0(h.ukb((w.OAb(0,10,w.UAb(V).length),w.UAb(V).substr(0,10)))),g1=I0(h.ukb((w.OAb(10,20,w.UAb(V).length),w.UAb(V).substr(10,10)))),q1=I0(h.ukb((w.OAb(20,30,w.UAb(V).length),w.UAb(V).substr(20,10)))),r=n.Tk(t.j,gt,-g1,-q1),J=h.ukb((w.OAb(31,34,w.UAb(V).length),w.UAb(V).substr(31,3))),h.ckb(J,"A")||h.ckb(J,"*")?n.vn(t.j,r,1,!0):h.ckb(J,"Q")?(At=B(Q.S7,H,6,1,15,1),At[0]=6,n.qn(t.j,r,At,!0)):(D=n.po(J,67),n.Cn(t.j,r,D)),it=n.ys(h.ukb((w.OAb(34,36,w.UAb(V).length),w.UAb(V).substr(34,2)))),it!=0&&n.tn(t.j,r,(n.Rk(),n.Pk)[n.Tl(t.j,r)]+it),y=n.ys(h.ukb((w.OAb(36,39,w.UAb(V).length),w.UAb(V).substr(36,3)))),y!=0&&(y==4?n.wn(t.j,r,32):n.jn(t.j,r,4-y)),ot=w.UAb(V).length<63?0:n.ys(h.ukb((w.OAb(60,63,w.UAb(V).length),w.UAb(V).substr(60,3)))),n.rn(t.j,r,ot,!1),F=w.UAb(V).length<45?0:n.ys(h.ukb((w.OAb(42,45,w.UAb(V).length),w.UAb(V).substr(42,3)))),F){case 0:break;case 1:n.vn(t.j,r,768,!0);break;case 2:n.vn(t.j,r,128,!0);break;case 3:n.vn(t.j,r,384,!0);break;default:n.vn(t.j,r,896,!0);}w.UAb(V).length>=48&&(w.PAb(47,w.UAb(V).length),w.UAb(V).charCodeAt(47)==49)&&n.vn(t.j,r,U0,!0),b1=w.UAb(V).length<51?0:n.ys(h.ukb((w.OAb(48,51,w.UAb(V).length),w.UAb(V).substr(48,3)))),b1!=0&&(f1==null&&(f1=B(Q.S7,H,6,ut,15,1)),f1[r]=b1);}for(E=0;E<Dt;E++){if((V=Ct.fhb(A))==null)return !1;P=h.Oib(h.ukb((w.OAb(0,3,w.UAb(V).length),w.UAb(V).substr(0,3))),e1,Ot)-1,i=h.Oib(h.ukb((w.OAb(3,6,w.UAb(V).length),w.UAb(V).substr(3,3))),e1,Ot)-1,l=h.Oib(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3))),e1,Ot),Jt=w.UAb(V).length<12?0:n.ys(h.ukb((w.OAb(9,12,w.UAb(V).length),w.UAb(V).substr(9,3)))),Bt=w.UAb(V).length<18?0:n.ys(h.ukb((w.OAb(15,18,w.UAb(V).length),w.UAb(V).substr(15,3)))),l==8&&(t.k||n.Lm(t.j,P)||n.Lm(t.j,i))&&(l=9),n.ds(t,P,i,l,Jt,Bt);}for(S=0;S<Qt;S++)if((V=Ct.fhb(A))==null)return !1;if((V=Ct.fhb(A))==null)return (t.i&1)!=0&&(t.f=n.lp(t.j)),n.js(t,f1),t.d||n.bw(t.j,15),!0;for(;V!=null&&!(h.ckb(V,vA)||h.ckb(V,le));){if(h.ckb(w.UAb(V).substr(0,6),"M  CHG")&&(O=h.Oib(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3))),e1,Ot),O>0))for(e=10,St=14,x=1;x<=O;++x,e+=8,St+=8)r=h.Oib(h.ukb((w.OAb(e,e+3,w.UAb(V).length),w.UAb(V).substr(e,e+3-e))),e1,Ot)-1,U=h.Oib(h.ukb((w.OAb(St,St+3,w.UAb(V).length),w.UAb(V).substr(St,St+3-St))),e1,Ot),n.jn(t.j,r,U);if(h.ckb(w.UAb(V).substr(0,6),"M  ISO")&&(O=h.Oib(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3))),e1,Ot),O>0))for(e=10,St=14,x=1;x<=O;++x,e+=8,St+=8)r=h.Oib(h.ukb((w.OAb(e,e+3,w.UAb(V).length),w.UAb(V).substr(e,e+3-e))),e1,Ot)-1,K=h.Oib(h.ukb((w.OAb(St,St+3,w.UAb(V).length),w.UAb(V).substr(St,St+3-St))),e1,Ot),n.tn(t.j,r,K);if(h.ckb(w.UAb(V).substr(0,6),"M  RAD")&&(O=h.Oib(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3))),e1,Ot),O>0))for(e=10,St=14,x=1;x<=O;++x,e+=8,St+=8)switch(r=h.Oib(h.ukb((w.OAb(e,e+3,w.UAb(V).length),w.UAb(V).substr(e,e+3-e))),e1,Ot)-1,wt=h.Oib(h.ukb((w.OAb(St,St+3,w.UAb(V).length),w.UAb(V).substr(St,St+3-St))),e1,Ot),wt){case 1:n.wn(t.j,r,16);break;case 2:n.wn(t.j,r,32);break;case 3:n.wn(t.j,r,48);}if((h.ckb(w.UAb(V).substr(0,6),"M  RBC")||h.ckb(w.UAb(V).substr(0,6),"M  RBD"))&&(O=h.Oib(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3))),e1,Ot),O>0))for(e=10,St=14,x=1;x<=O;++x,e+=8,St+=8)switch(r=h.Oib(h.ukb((w.OAb(e,e+3,w.UAb(V).length),w.UAb(V).substr(e,e+3-e))),e1,Ot)-1,Vt=h.Oib(h.ukb((w.OAb(St,St+3,w.UAb(V).length),w.UAb(V).substr(St,St+3-St))),e1,Ot),Vt){case 3:case -1:n.vn(t.j,r,112,!0);break;case 1:n.vn(t.j,r,8,!0);break;case 2:n.vn(t.j,r,104,!0);break;case 4:n.vn(t.j,r,56,!0);}if(h.ckb(w.UAb(V).substr(0,6),"M  ALS")&&(r=h.Oib(h.ukb((w.OAb(7,10,w.UAb(V).length),w.UAb(V).substr(7,3))),e1,Ot)-1,r>=0)){for(bt=h.Oib(h.ukb((w.OAb(10,13,w.UAb(V).length),w.UAb(V).substr(10,3))),e1,Ot),u=(w.PAb(14,w.UAb(V).length),w.UAb(V).charCodeAt(14)==84),b1=B(Q.S7,H,6,bt,15,1),e=16,x=0;x<bt;++x,e+=4)dt=h.ukb((w.OAb(e,e+4,w.UAb(V).length),w.UAb(V).substr(e,e+4-e))),b1[x]=n.po(dt,1);n.Cn(t.j,r,6),n.qn(t.j,r,b1,u);}if(h.ckb(w.UAb(V).substr(0,6),"M  SUB")&&(O=h.Oib(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3))),e1,Ot),O>0)){for(e=10,St=14,x=1;x<=O;++x,e+=8,St+=8)if(r=h.Oib(h.ukb((w.OAb(e,e+3,w.UAb(V).length),w.UAb(V).substr(e,e+3-e))),e1,Ot)-1,Nt=h.Oib(h.ukb((w.OAb(St,St+3,w.UAb(V).length),w.UAb(V).substr(St,St+3-St))),e1,Ot),Nt==-2)n.vn(t.j,r,B1,!0);else if(Nt>0){for(Gt=0,o=0;o<t.j.r;o++)(n.Zl(t.j,0,o)==r||n.Zl(t.j,1,o)==r)&&++Gt;Nt>Gt&&n.vn(t.j,r,Z1,!0);}}if(h.ckb(w.UAb(V).substr(0,6),"M  RGP")&&(O=h.Oib(h.ukb((w.OAb(6,9,w.UAb(V).length),w.UAb(V).substr(6,3))),e1,Ot),O>0))for(e=10,St=14,x=1;x<=O;++x,e+=8,St+=8)r=h.Oib(h.ukb((w.OAb(e,e+3,w.UAb(V).length),w.UAb(V).substr(e,e+3-e))),e1,Ot)-1,Et=h.Oib(h.ukb((w.OAb(St,St+3,w.UAb(V).length),w.UAb(V).substr(St,St+3-St))),e1,Ot),Et>=1&&Et<=20&&n.Cn(t.j,r,n.po("R"+Et,2));(h.ckb(w.UAb(V).substr(0,6),"M  SAL")&&w.UAb(V).length>=17||h.ckb(w.UAb(V).substr(0,6),"M  SDT")&&w.UAb(V).length>=12||h.ckb(w.UAb(V).substr(0,6),"M  SED")&&w.UAb(V).length>=12)&&(!b&&(b=new s.Zwb()),Ht=(w.OAb(7,10,w.UAb(V).length),w.UAb(V).substr(7,3)),T=(G=Ht,s.Znb(s.Mwb(b,G))),T||(T=new n.Es(),s.Rwb(b,Ht,T)),h.ckb(w.UAb(V).substr(0,6),"M  SAL")&&h.ckb(w.UAb(V).substr(10,w.UAb("  1").length),"  1")?T.a=h.Oib(h.ukb((w.OAb(13,17,w.UAb(V).length),w.UAb(V).substr(13,4))),e1,Ot):h.ckb(w.UAb(V).substr(0,6),"M  SDT")?T.b=h.rkb((w.PAb(11,w.UAb(V).length+1),w.UAb(V).substr(11)),ir):h.ckb(w.UAb(V).substr(0,6),"M  SED")&&(T.c=h.ukb((w.PAb(11,w.UAb(V).length+1),w.UAb(V).substr(11))))),V=Ct.fhb(A);}}catch(U1){if(U1=M1(U1),Zt(U1,19))return g=U1,h.MB(g,(h.Rkb(),h.Qkb),""),false;throw Lt(U1);}if(b)for(c=(st=new s.Gpb(b).a.cd().Tb(),new s.Npb(st));c.a.ed();)T=(k=c.a.fd(),k.jd()),T.b&&T.a!=0&&T.c!=null&&n.mn(t.j,T.a-1,T.c);return t.e&&(n.vs(t),n.es(t)),(t.i&1)!=0&&(t.f=n.lp(t.j)),n.js(t,f1),n.bw(t.j,15),true;},n.Bs=function(t,A){var e,r,P,i,D;for(t.b&&s.Kwb(t.b),t.c&&s.Kwb(t.c),P=0,r=Ct.fhb(A);r!=null&&h.ckb(w.UAb(r).substr(0,7),Ue);){for(r=h.ukb((w.PAb(7,w.UAb(r).length+1),w.UAb(r).substr(7)));D=w.UAb("-").length,h.ckb(w.UAb(r).substr(w.UAb(r).length-D,D),"-");){if(e=Ct.fhb(A),!h.ckb(w.UAb(e).substr(0,7),Ue))return  false;r=h.ukb(h.$jb(h.tkb(r,0,w.UAb(r).length-1),(w.PAb(7,w.UAb(e).length+1),w.UAb(e).substr(7))));}if(h.ckb(w.UAb(r).substr(0,5),"BEGIN")){if(i=h.ukb((w.PAb(6,w.UAb(r).length+1),w.UAb(r).substr(6))),h.ckb(w.UAb(i).substr(0,4),"CTAB"))P=1;else if(h.ckb(w.UAb(i).substr(0,4),"ATOM"))P=2;else if(h.ckb(w.UAb(i).substr(0,4),"BOND"))P=3;else if(h.ckb(w.UAb(i).substr(0,6),"SGROUP"))P=4;else if(h.ckb(w.UAb(i).substr(0,10),"COLLECTION"))P=5;else return  false;}else if(h.ckb(w.UAb(r).substr(0,3),"END"))P=0;else if(P==1)n.ss(t,r);else if(P==2)n.os(t,r);else if(P==3)n.qs(t,r);else if(P==4)n.us(t,r);else if(P==5)n.rs(t,r);else return  false;r=Ct.fhb(A);}for(;r!=null&&!(h.ckb(w.UAb(r).substr(0,6),vA)||h.ckb(r,le));)r=Ct.fhb(A);return  true;},n.Cs=function(){this.i=0;},n.Ds=function(){this.i=1;},I(127,1,{},n.Cs,n.Ds),L.a=false,L.d=false,L.e=false,L.g=false,L.i=0,L.k=false,Q.B8=N(127),n.Es=function(){},I(224,1,{224:1},n.Es),L.a=0,L.b=false,Q.A8=N(224),n.Fs=function(t){},n.Gs=function(t,A){var e,r;for(r=Q0.Fnb(t.a,A),e=w.UAb(r).length;e<10;e++)h.Gkb(t.b,32);h.Kkb(t.b,r);},n.Hs=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt;for(At=(h.Rkb(),m1),h.Kkb(t.b,"M  V30 BEGIN CTAB"+At),h.Kkb(t.b,Z8+A.q+" "+A.r+" 0 0 0"+At),h.Kkb(t.b,"M  V30 BEGIN ATOM"+At),o=0;o<A.q;o++){if(h.Kkb(t.b,Ue+(o+1)),(A.v==null?null:A.v[o])!=null){for(T=A.v==null?null:A.v[o],ot=Rt(z(A.B[o],1),0),h.Kkb(t.b,ot?" NOT[":" ["),x=0;x<T.length;x++)switch(x>0&&h.Kkb(t.b,","),G=(n.Rk(),n.Lk)[T[x]],w.UAb(G).length){case 2:case 3:case 1:h.Kkb(t.b,G);break;default:h.Kkb(t.b,"?");}h.Kkb(t.b,"]");}else Rt(z(A.B[o],1),0)?h.Kkb(t.b," A"):A.C[o]>=129&&A.C[o]<=144||A.C[o]==154?h.Kkb(t.b," R#"):h.Kkb(t.b," "+(n.Rk(),n.Lk)[A.C[o]]);if(e?(h.Kkb(t.b," "+pt(Z0*t.c*A.J[o].a)/Z0),h.Kkb(t.b," "+pt(Z0*t.c*-A.J[o].b)/Z0),h.Kkb(t.b," "+pt(Z0*t.c*-A.J[o].c)/Z0)):h.Kkb(t.b," 0 0 0"),h.Kkb(t.b," "+d.Math.abs(A.w[o])),A.s[o]!=0&&h.Kkb(t.b," CHG="+A.s[o]),(A.u[o]&48)!=0)switch(h.Kkb(t.b," RAD="),A.u[o]&48){case 16:h.Kkb(t.b,"1");break;case 32:h.Kkb(t.b,"2");break;case 48:h.Kkb(t.b,"3");}((A.u[o]&3)==1||(A.u[o]&3)==2)&&(h.Kkb(t.b," CFG="),(A.u[o]&3)==1?h.Kkb(t.b,"1"):h.Kkb(t.b,"2")),A.A[o]!=0&&h.Kkb(t.b," MASS="+A.A[o]),Vt=((A.u[o]&bA)>>>27)-1,Vt!=-1&&h.Kkb(t.b," VAL="+(Vt==0?"-1":h.sjb(Vt))),c=A.C[o],(c>=129&&c<=144||c==154)&&h.Kkb(t.b," RGROUPS=(1 "+(c==154?0:c>=142?c-141:c-125)+")"),O=z(X0,A.B[o]),nt(O,384)==0?h.Kkb(t.b," HCOUNT=2"):nt(O,128)==0?h.Kkb(t.b," HCOUNT=1"):nt(O,1792)==0?h.Kkb(t.b," HCOUNT=-1"):nt(O,1664)==0&&h.Kkb(t.b," HCOUNT=1"),st=z(A.B[o],6144),nt(st,0)!=0&&(Rt(z(st,Z1),0)?h.Kkb(t.b," SUBST="+(A.e[o]+1)):h.Kkb(t.b," SUBST=-1")),Dt=z(A.B[o],120),nt(Dt,0)!=0&&(nt(Dt,112)==0?h.Kkb(t.b," RBCNT=-1"):nt(Dt,8)==0||nt(Dt,104)==0?h.Kkb(t.b," RBCNT=2"):nt(Dt,88)==0?h.Kkb(t.b," RBCNT=3"):nt(Dt,56)==0&&h.Kkb(t.b," RBCNT=4")),h.Kkb(t.b,At);}for(h.Kkb(t.b,"M  V30 END ATOM"+At),h.Kkb(t.b,"M  V30 BEGIN BOND"+At),g=0;g<A.r;g++){switch(h.Kkb(t.b,Ue+(g+1)),A.H[g]){case 1:lt=1,bt=0;break;case 2:lt=2,bt=0;break;case 4:lt=3,bt=0;break;case 129:lt=1,bt=3;break;case 257:lt=1,bt=1;break;case 386:lt=2,bt=2;break;case 8:lt=4,bt=0;break;case 16:lt=9,bt=0;break;default:lt=1,bt=0;}F=A.G[g]&31,F!=0&&(F==8?lt=4:F==3?lt=5:F==9?lt=6:F==10?lt=7:lt=8),h.Kkb(t.b," "+lt+" "+(A.D[0][g]+1)+" "+(A.D[1][g]+1)),bt!=0&&h.Kkb(t.b," CFG="+bt),Qt=A.G[g]&384,wt=Qt==0?0:Qt==256?1:2,wt!=0&&h.Kkb(t.b," TOPO="+wt),h.Kkb(t.b,At);}for(h.Kkb(t.b,"M  V30 END BOND"+At),R=false,l=0;l<A.q;l++)if((A.t==null||A.t[l]==null?null:h.akb((J=A.t[l],O1.Enb(),J)))!=null){R=true;break;}if(R){for(h.Kkb(t.b,"M  V30 BEGIN SGROUP"+At),S=0,U=0;U<A.q;U++)(A.t==null||A.t[U]==null?null:h.akb((V=A.t[U],O1.Enb(),V)))!=null&&(++S,h.Kkb(t.b,Ue+S+" DAT "+S+" ATOMS=(1 "+(U+1)+') FIELDNAME="'+ir+'" -'+At),h.Kkb(t.b,'M  V30 FIELDDISP="'),n.Gs(t,e?t.c*A.J[U].a:0),n.Gs(t,e?t.c*A.J[U].b:0),h.Kkb(t.b,'    DA    ALL  1       5" FIELDDATA="'+(A.t==null||A.t[U]==null?null:h.akb((J=A.t[U],O1.Enb(),J)))+'"'+At));h.Kkb(t.b,"M  V30 END SGROUP"+At);}for(ut=false,r=0,K=B(Q.S7,H,6,32,15,1),i=B(Q.S7,H,6,32,15,1),y=0;y<A.f;y++)((A.u[y]&3)==1||(A.u[y]&3)==2)&&(ut=true,ct=(A.u[y]&l1)>>18,ct==1?++i[(A.u[y]&l1)>>18!=1&&(A.u[y]&l1)>>18!=2?-1:(A.u[y]&v0)>>20]:ct==2?++K[(A.u[y]&l1)>>18!=1&&(A.u[y]&l1)>>18!=2?-1:(A.u[y]&v0)>>20]:++r);for(P=0,it=B(Q.S7,H,6,32,15,1),D=B(Q.S7,H,6,32,15,1),k=0;k<A.g;k++)n.fm(A,k)!=2&&((A.F[k]&3)==1||(A.F[k]&3)==2)&&(ut=true,ct=(A.F[k]&768)>>8,ct==1?++D[(A.F[k]&768)>>8!=1&&(A.F[k]&768)>>8!=2?-1:(A.F[k]&E0)>>10]:ct==2?++it[(A.F[k]&768)>>8!=1&&(A.F[k]&768)>>8!=2?-1:(A.F[k]&E0)>>10]:++P);if(ut){if(h.Kkb(t.b,"M  V30 BEGIN COLLECTION"+At),r!=0){for(h.Kkb(t.b,"M  V30 MDLV30/STEABS ATOMS=("+r),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&l1)>>18==0&&h.Kkb(t.b," "+(u+1));h.Kkb(t.b,")"+At);}if(P!=0){for(h.Kkb(t.b,"M  V30 MDLV30/STEABS BONDS=("+P),b=0;b<A.g;b++)n.fm(A,b)!=2&&((A.F[b]&3)==1||(A.F[b]&3)==2)&&(A.F[b]&768)>>8==0&&h.Kkb(t.b," "+(b+1));h.Kkb(t.b,")"+At);}for(E=0;E<32;E++){if(K[E]!=0){for(h.Kkb(t.b,Y8+(E+1)+" ATOMS=("+K[E]),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&l1)>>18==2&&((A.u[u]&l1)>>18!=1&&(A.u[u]&l1)>>18!=2?-1:(A.u[u]&v0)>>20)==E&&h.Kkb(t.b," "+(u+1));h.Kkb(t.b,")"+At);}if(i[E]!=0){for(h.Kkb(t.b,X8+(E+1)+" ATOMS=("+i[E]),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&l1)>>18==1&&((A.u[u]&l1)>>18!=1&&(A.u[u]&l1)>>18!=2?-1:(A.u[u]&v0)>>20)==E&&h.Kkb(t.b," "+(u+1));h.Kkb(t.b,")"+At);}if(it[E]!=0){for(h.Kkb(t.b,Y8+(E+1)+" BONDS=("+it[E]),b=0;b<A.g;b++)n.fm(A,b)!=2&&((A.F[b]&3)==1||(A.F[b]&3)==2)&&(A.F[b]&768)>>8==2&&((A.F[b]&768)>>8!=1&&(A.F[b]&768)>>8!=2?-1:(A.F[b]&E0)>>10)==E&&h.Kkb(t.b," "+(b+1));h.Kkb(t.b,")"+At);}if(D[E]!=0){for(h.Kkb(t.b,X8+(E+1)+" BONDS=("+D[E]),b=0;b<A.g;b++)n.fm(A,b)!=2&&((A.F[b]&3)==1||(A.F[b]&3)==2)&&(A.F[b]&768)>>8==1&&((A.F[b]&768)>>8!=1&&(A.F[b]&768)>>8!=2?-1:(A.F[b]&E0)>>10)==E&&h.Kkb(t.b," "+(b+1));h.Kkb(t.b,")"+At);}}h.Kkb(t.b,"M  V30 END COLLECTION"+At);}h.Kkb(t.b,UP+At);},n.Is=function(){n.Fs(this),this.b=new h.Nkb();},n.Js=function(t){n.Ks.call(this,t);},n.Ks=function(t){n.Ls.call(this,t,new h.Nkb());},n.Ls=function(t,A){var e,r,P,i;n.Fs(this),this.a=new Q0.Knb((r=Pr,r)),n.bw(t,15),i=(h.Rkb(),m1),this.b=A,P=t.P!=null?t.P:"",h.Kkb(this.b,P+i),h.Kkb(this.b,"Actelion Java MolfileCreator 2.0"+i+i),h.Kkb(this.b,"  0  0  0  0  0  0              0 V3000"+i),this.c=1,e=n.Ns(t),e&&(this.c=n.Ms(t)),n.Hs(this,t,e),h.Kkb(this.b,vA+i);},n.Ms=function(t){var A,e,r,P,i,D,u,o,l;if(l=1,r=n.Wl(t,t.q,t.r,(n.Rk(),n.Qk)),r!=0)(r<1||r>3)&&(l=1.5/r);else {for(o=_8,A=1;A<t.q;A++)for(e=0;e<A;e++)i=t.J[e].a-t.J[A].a,D=t.J[e].b-t.J[A].b,u=t.J[e].c-t.J[A].c,P=i*i+D*D+u*u,o>P&&(o=P);l=3/d.Math.max(.75,o);}return l;},n.Ns=function(t){var A;if(t.q==1)return  true;for(A=1;A<t.q;A++)if(t.J[A].a!=t.J[0].a||t.J[A].b!=t.J[0].b||t.J[A].c!=t.J[0].c)return  true;return  false;},n.Os=function(t,A){var e;return e=new n.Is(),e.c=A,n.bw(t,15),n.Hs(e,t,true),e.b.a;},I(150,1,{},n.Is,n.Js),L.c=1,Q.C8=N(150),n.Rs=function(){n.Rs=i1,n.Ps=j(v(Q.Qcb,2),b0,13,0,[j(v(Q.Qcb,1),yt,2,6,["Acyl","gCaHA`AIf`@"]),j(v(Q.Qcb,1),yt,2,6,["Alloc","gNph@l@ILzuR@@"]),j(v(Q.Qcb,1),yt,2,6,["Allyl","Al"]),j(v(Q.Qcb,1),yt,2,6,["Benzyl;Bn","daD@`F@DjUZxHH@@"]),j(v(Q.Qcb,1),yt,2,6,["Boc","daxD`@S@AIgijj@@"]),j(v(Q.Qcb,1),yt,2,6,["BOM;BzOM","deTH`@H@Re[TYj`@@@"]),j(v(Q.Qcb,1),yt,2,6,["Bs","dmtDpAdLS`aPAIe]jf`@e`@@"]),j(v(Q.Qcb,1),yt,2,6,["Bt","dew@`@aJ@DiY]paej`@@@"]),j(v(Q.Qcb,1),yt,2,6,["Btm","did@P@BJ@Die_ahH@@@"]),j(v(Q.Qcb,1),yt,2,6,["Butyl;nButyl","gJPHB@IRuP@"]),j(v(Q.Qcb,1),yt,2,6,["Benzoyl;Bz","didH`@p@RYm^Eh@@@@"]),j(v(Q.Qcb,1),yt,2,6,["Bzh","dg|@`N@LdbRbtJUB]aAP@@@@"]),j(v(Q.Qcb,1),yt,2,6,["Cbz","dmtD`@S@AIgYVUZh@@@@"]),j(v(Q.Qcb,1),yt,2,6,["cButyl","gKPHL@IThuT@@"]),j(v(Q.Qcb,1),yt,2,6,["cHeptyl","daD@`L@DjWVzjj`@"]),j(v(Q.Qcb,1),yt,2,6,["cHexyl","gOpHL@IToWUU@@"]),j(v(Q.Qcb,1),yt,2,6,["cOctyl","did@`L@DjWWajjj@@"]),j(v(Q.Qcb,1),yt,2,6,["cPentyl","gFpHL@ITimUP@"]),j(v(Q.Qcb,1),yt,2,6,["cPropyl","gBPHL@Qxjh@"]),j(v(Q.Qcb,1),yt,2,6,["COOH","gC`h@l@ILt@@ !Bb@K~@Hc}"]),j(v(Q.Qcb,1),yt,2,6,["DEAE","daz@`@x@RiUjj`@"]),j(v(Q.Qcb,1),yt,2,6,["DEIPS","diD@P@\\B@DjfVjj`@"]),j(v(Q.Qcb,1),yt,2,6,["DMIPS","gNpD@xD@RjZjh@"]),j(v(Q.Qcb,1),yt,2,6,["DMPM","dcLD`@kPCIEMDdcttDDT@@"]),j(v(Q.Qcb,1),yt,2,6,["DMPS","deT@P@\\B@LddTjPsU@@@@"]),j(v(Q.Qcb,1),yt,2,6,["DMTr","fak@b@@Mt@ISZ{SMjo{NQKfm@AU@@@E@@@@"]),j(v(Q.Qcb,1),yt,2,6,["DNP","dkmB`hdDt~@HeNfS{HihheCAUhBHX@@"]),j(v(Q.Qcb,1),yt,2,6,["DNS;Dan","fhi`a@KPP@HH@YIHYheEhYKQgKP@@QP@@@"]),j(v(Q.Qcb,1),yt,2,6,["DPIPS","fdyAA@H@\\B@FRRIQSQIHzp_Qjh@h@@@@@"]),j(v(Q.Qcb,1),yt,2,6,["DPTBS","fleAA@H@\\B@FRRIQSRIIWNbEMU@EP@@@@@"]),j(v(Q.Qcb,1),yt,2,6,["DTBMS","dmT@P@\\B@Djffjjjh@@"]),j(v(Q.Qcb,1),yt,2,6,["Ethyl","eMBD@ch@"]),j(v(Q.Qcb,1),yt,2,6,["Fmoc","fde@b@@Hp@IL{LrjxeVCzKUT@@@P@@@"]),j(v(Q.Qcb,1),yt,2,6,["iAmyl","gGPHJ@YIDZj@@"]),j(v(Q.Qcb,1),yt,2,6,["iButyl","gJPHB@ITuP@"]),j(v(Q.Qcb,1),yt,2,6,["Im","gFtHAj@IRnKSP@"]),j(v(Q.Qcb,1),yt,2,6,["iPropyl","gC`HL@Qz`@"]),j(v(Q.Qcb,1),yt,2,6,["MDIPS","diD@P@\\B@DjfZjj`@"]),j(v(Q.Qcb,1),yt,2,6,["MDPS","foA@A@@NA@CIIEEBdeeVLzj@@@@@@"]),j(v(Q.Qcb,1),yt,2,6,["Methyl","eFBH@c@@"]),j(v(Q.Qcb,1),yt,2,6,["MEM","gNphAR@IRoUT@@"]),j(v(Q.Qcb,1),yt,2,6,["Mes","deT@`J@DjY{[`bB`@@"]),j(v(Q.Qcb,1),yt,2,6,["MMTr","ffcAB@@Z@Dim]ifuWYrI\\uh@Jh@@@@@@"]),j(v(Q.Qcb,1),yt,2,6,["MOM","gCaHA`AJZ`@"]),j(v(Q.Qcb,1),yt,2,6,["MPM;PMB","deTH`@d@Rfuunh@J@@"]),j(v(Q.Qcb,1),yt,2,6,["Ms","gJPdH`DD@cuh@"]),j(v(Q.Qcb,1),yt,2,6,["MTM","gC`D@DX@Rfh@"]),j(v(Q.Qcb,1),yt,2,6,["mTolyl","daD@`N@DjWjXHB@@"]),j(v(Q.Qcb,1),yt,2,6,["N3","gClHaE`@RnReX@"]),j(v(Q.Qcb,1),yt,2,6,["nAmyl;Amyl","gGPHJ@IRmU@@"]),j(v(Q.Qcb,1),yt,2,6,["neoAm","gGPHJ@IUMU@@"]),j(v(Q.Qcb,1),yt,2,6,["NO2,Nitro","gChhhE`BRnRYh@"]),j(v(Q.Qcb,1),yt,2,6,["Np","deVDaHAI@HeNR[e_aZ@B@@"]),j(v(Q.Qcb,1),yt,2,6,["nPropyl;Propyl","gC`HL@IST@@"]),j(v(Q.Qcb,1),yt,2,6,["oTolyl","daD@`J@DjYvxH`@@"]),j(v(Q.Qcb,1),yt,2,6,["Phenyl","gOpHL@IToVD@@@"]),j(v(Q.Qcb,1),yt,2,6,["Pht","dcLL`@RU@Dfyed]ZBA`@@"]),j(v(Q.Qcb,1),yt,2,6,["Piv;Pv","gNqHA`AIffj`@"]),j(v(Q.Qcb,1),yt,2,6,["PMBM","dcLD`@T`AJUm]FZh@J@@"]),j(v(Q.Qcb,1),yt,2,6,["PNB","dcNLaHAEt@bTyInUvxV`@f@@"]),j(v(Q.Qcb,1),yt,2,6,["Poc","didD`@S@AIgexVjj`@"]),j(v(Q.Qcb,1),yt,2,6,["PPi","diDFsHSB[`|J|A@Lxn{lddqdZih@@"]),j(v(Q.Qcb,1),yt,2,6,["pTolyl","daD@`N@DjWzXHB@@"]),j(v(Q.Qcb,1),yt,2,6,["sAmyl","gGPHL@YIDZj@@"]),j(v(Q.Qcb,1),yt,2,6,["sButyl","gJPHL@ITuP@"]),j(v(Q.Qcb,1),yt,2,6,["SEM","diDHPFApD@rRQUJjj`@"]),j(v(Q.Qcb,1),yt,2,6,["SES","dedDpHP@``AgCIICeHmUT@@"]),j(v(Q.Qcb,1),yt,2,6,["SO3H","gJQdHl@``D^m@@"]),j(v(Q.Qcb,1),yt,2,6,["tAmyl","gGPHB@IUMU@@"]),j(v(Q.Qcb,1),yt,2,6,["TBDMS;TBS","dax@P@\\B@Djfjjh@@"]),j(v(Q.Qcb,1),yt,2,6,["TBDPS","fdy@A@@NA@CIIEEEIde]XOhuPAT@@@@@"]),j(v(Q.Qcb,1),yt,2,6,["TBMPS","dg\\HPHApH@rRQJJPjg]UAT@@@"]),j(v(Q.Qcb,1),yt,2,6,["tButyl,tBu","gJPHB@Q}T@@"]),j(v(Q.Qcb,1),yt,2,6,["TDS","ded@P@\\B@LddTeeUUP@@"]),j(v(Q.Qcb,1),yt,2,6,["Tf","daxDhHP@``BiAiCiCIICHmU@@"]),j(v(Q.Qcb,1),yt,2,6,["TFA","gNqBJIARFdF@YEHYUL@@"]),j(v(Q.Qcb,1),yt,2,6,["Thexyl","gNpHB@IUMUT@@"]),j(v(Q.Qcb,1),yt,2,6,["THF","gFqH@PAJYujj@@"]),j(v(Q.Qcb,1),yt,2,6,["THP","gOqH@PAJYZzjh@"]),j(v(Q.Qcb,1),yt,2,6,["TIPS","dmT@P@\\B@DjfYjjjh@@"]),j(v(Q.Qcb,1),yt,2,6,["TMS","gJPD@xD@czh@"]),j(v(Q.Qcb,1),yt,2,6,["Tosyl;Ts","dmtDPHP@``CIICLeaeZ@B@@"]),j(v(Q.Qcb,1),yt,2,6,["Troc","diDDHJxHaHcH`PCHiBeJjf@@"]),j(v(Q.Qcb,1),yt,2,6,["Trt","fbm@B@A@FRQIRKQPiIZdoIcdHJ`@@@@@@"]),j(v(Q.Qcb,1),yt,2,6,["Xyl","did@`J@DjYynBHH@@"])]);},n.Ss=function(){n.Rs();var t,A,e,r,P,i,D,u,o;for(n.Qs=new s.Zwb(),D=n.Ps,u=0,o=D.length;u<o;++u)for(i=D[u],P=h.qkb(i[0],";"),A=P,e=0,r=A.length;e<r;++e)t=A[e],s.Rwb(n.Qs,h.okb(w.UAb(t).toLowerCase(),"-"),i[1]);},n.Ts=function(t){n.Rs();var A,e,r,P,i,D,u,o,l,U;if(t==null||w.UAb(t).length==0)return null;for(D=null,t=h.okb(w.UAb(t).toLowerCase(),"-"),o=n.Ps,l=0,U=o.length;l<U;++l)for(u=o[l],i=h.qkb(u[0],";"),e=i,r=0,P=e.length;r<P;++r){if(A=e[r],h.ckb(h.okb(w.UAb(A).toLowerCase(),"-"),t))return A;if(h.rkb(h.okb(w.UAb(A).toLowerCase(),"-"),t))if(D==null)D=A;else return "";}return D;},n.Qs=null,n.Ws=function(){var t,A,e,r;for(this.b=new s.rub(),this.c=new s.rub(),this.d=new s.rub(),n.Us=j(v(Q.o8,1),C1,18,0,[new n.mk(1,"Hydrogen","H",2.2),new n.mk(2,"Helium","He",0),new n.mk(3,"Lithium","Li",.98),new n.mk(4,"Beryllium","Be",1.57),new n.mk(5,"Boron","B",2.04),new n.mk(6,"Carbon","C",2.55),new n.mk(7,"Nitrogen","N",3.04),new n.mk(8,"Oxygen","O",3.44),new n.mk(9,"Fluorine","F",3.98),new n.mk(10,"Neon","Ne",0),new n.mk(11,"Sodium","Na",.93),new n.mk(12,"Magnesium","Mg",1.31),new n.mk(13,"Aluminum","Al",1.61),new n.mk(14,"Silicon","Si",1.9),new n.mk(15,"Phosphorus","P",2.19),new n.mk(16,"Sulfur","S",2.58),new n.mk(17,"Chlorine","Cl",3.16),new n.mk(18,"Argon","Ar",0),new n.mk(19,"Potassium","K",1),new n.mk(20,"Calcium","Ca",1.36),new n.mk(21,"Scandium","Sc",1.54),new n.mk(22,"Titanium","Ti",1.63),new n.mk(23,"Vanadium","V",1.66),new n.mk(24,"Chromium","Cr",1.55),new n.mk(25,"Manganese","Mn",1.83),new n.mk(26,"Iron","Fe",1.88),new n.mk(27,"Cobalt","Co",1.91),new n.mk(28,"Nickel","Ni",1.9),new n.mk(29,"Copper","Cu",1.65),new n.mk(30,"Zinc","Zn",1.81),new n.mk(31,"Gallium","Ga",2.01),new n.mk(32,"Germanium","Ge",0),new n.mk(33,"Arsenic","As",2.18),new n.mk(34,"Selenium","Se",2.55),new n.mk(35,"Bromine","Br",0),new n.mk(36,"Krypton","Kr",2.96),new n.mk(37,"Rubidium","Rb",0),new n.mk(38,"Strontium","Sr",.82),new n.mk(39,"Yttrium","Y",.95),new n.mk(40,"Zirconium","Zr",1.22),new n.mk(41,"Niobium","Nb",1.33),new n.mk(42,"Molybdenum","Mo",1.6),new n.mk(43,"Technetium","Tc",2.16),new n.mk(44,"Ruthenium","Ru",1.9),new n.mk(45,"Rhodium","Rh",2.2),new n.mk(46,"Palladium","Pd",2.28),new n.mk(47,"Silver","Ag",2.2),new n.mk(48,"Cadmium","Cd",1.93),new n.mk(49,"Indium","In",1.69),new n.mk(50,"Tin","Sn",1.78),new n.mk(51,"Antimony","Sb",1.96),new n.mk(52,"Tellurium","Te",0),new n.mk(53,"Iodine","I",2.05),new n.mk(54,"Xenon","Xe",2.1),new n.mk(55,"Cesium","Cs",2.66),new n.mk(56,"Barium","Ba",2.6),new n.mk(57,"Lanthanum","La",.79),new n.mk(58,"Cerium","Ce",.89),new n.mk(59,"Praesodymium","Pr",1.1),new n.mk(60,"Neodymium","Nd",1.12),new n.mk(61,"Promethium","Pm",1.13),new n.mk(62,"Samarium","Sm",1.14),new n.mk(63,"Europium","Eu",0),new n.mk(64,"Gadolinium","Gd",1.17),new n.mk(65,"Terbium","Tb",0),new n.mk(66,"Dyprosium","Dy",1.2),new n.mk(67,"Holmium","Ho",0),new n.mk(68,"Erbium","Er",1.22),new n.mk(69,"Thulium","Tm",1.23),new n.mk(70,"Ytterbium","Yb",1.24),new n.mk(71,"Lutetium","Lu",1.25),new n.mk(72,"Hafnium","Hf",0),new n.mk(73,"Tantalium","Ta",1.27),new n.mk(74,"Wolfram","W",1.3),new n.mk(75,"Rhenium","Re",1.5),new n.mk(76,"Osmium","Os",2.36),new n.mk(77,"Iridium","Ir",1.9),new n.mk(78,"Platinum","Pt",2.2),new n.mk(79,"Gold","Au",2.2),new n.mk(80,"Mercury","Hg",2.28),new n.mk(81,"Thallium","Tl",2.54),new n.mk(82,"Lead","Pb",2),new n.mk(83,"Bismuth","Bi",1.62),new n.mk(84,"Polonium","Po",2.33),new n.mk(85,"Astatine","At",2.02),new n.mk(86,"Radon","Rn",2),new n.mk(87,"Francium","Fr",2.2),new n.mk(88,"Radium","Ra",0),new n.mk(89,"Actinium","Ac",.7),new n.mk(90,"Thorium","Th",.89),new n.mk(91,"Protactinium","Pa",1.1),new n.mk(92,"Uranium","U",1.3),new n.mk(93,"Neptunium","Np",1.5),new n.mk(94,"Plutonium","Pu",1.38),new n.mk(95,"Americium","Am",1.36),new n.mk(96,"Curium","Cm",1.28),new n.mk(97,"Berkelium","Bk",1.3),new n.mk(98,"Californium","Cf",1.3),new n.mk(99,"Einsteinium","E",1.3),new n.mk(100,"Fermium","Fm",1.3),new n.mk(101,"Mendelevium","Mv",1.3)]),A=n.Us,e=0,r=A.length;e<r;++e)t=A[e],s.sob(this.b,h.sjb(t.c),t),s.tob(this.c,t.b,t),s.tob(this.d,w.UAb(t.d).toLowerCase(),t);this.a=B(Q.xfb,Tt,6,n.Us.length,16,1),this.a[3]=true,this.a[11]=true,this.a[19]=true,this.a[37]=true,this.a[55]=true,this.a[87]=true;},n.Xs=function(t){return s.pob((!n.Vs&&(n.Vs=new n.Ws()),n.Vs).b,h.sjb(t));},I(401,1,{},n.Ws),L.pb=function(){return "";},Q.D8=N(401),n.Zs=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g;for(l=t.i.M,U=0,i=0;i<e;i++)l>A[i]&&(l=A[i],U=i);for(g=B(Q.S7,H,6,e,15,1),o=U>0?U-1:e-1,y=U<e-1?U+1:0,D=A[o]<A[y],P=0;P<e;P++)g[P]=A[U],D?--U<0&&(U=e-1):++U==e&&(U=0);for(b=new s.trb(t.j);b.a<b.c.a.length;)if(c=s.srb(b),c.length==e){for(r=true,u=0;u<e;u++)if(c[u]!=g[u]){r=false;break;}if(r)return;}s.sj(t.j,g),T=n.jt(t,g),s.sj(t.k,T),n.Bt(t,g,T);},n.$s=function(t,A,e){var r,P,i,D,u,o,l;for(o=B(Q.S7,H,6,t.g,15,1),D=B(Q.S7,H,6,t.g,15,1),l=B(Q.xfb,Tt,6,t.i.f,16,1),r=n.Zl(t.i,0,A),P=n.Zl(t.i,1,A),o[0]=r,o[1]=P,D[1]=-1,l[P]=true,u=1;u>=1;){if(++D[u],D[u]==n.$o(t.i,o[u])){l[o[u]]=false,--u;continue;}if(i=n.Zo(t.i,o[u],D[u]),!(l[i]||e[i])){if(i==r&&u>1){if(n.Zs(t,o,u+1),t.j.a.length>=f0)return;continue;}u+1<t.g&&(++u,o[u]=i,l[i]=true,D[u]=-1);}}},n._s=function(t,A,e,r,P,i,D,u){var o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct;for(lt=s.xj(t.j,A),y=lt,T=0,c=y.length;T<c;++T)if(U=y[T],!n.xt(t,U))return  true;for(ut=s.xj(t.k,A),Dt=ut.length,g=0,l=0,ct=false,G=0;G<Dt;G++)g<<=1,l<<=1,n.zt(t,ut[G])?g|=1:u&&n.wt(t,ut[G])?(g|=1,l|=1):(o=e[A][G],o!=-1&&(r[o]?P[o]&&(g|=1,i[o]||(l|=1)):ct=true));switch(O=false,Dt){case 3:for(k=j(v(Q.S7,1),H,6,15,[2,1,4]),O=true,E=0;E<3;E++)(g&k[E])==k[E]&&(n.Tl(t.i,lt[E])==6&&n.Bl(t.i,lt[E])==1||n.Tl(t.i,lt[E])==5&&n.Bl(t.i,lt[E])==0)&&(P[A]=true,D[A]=E,(l&k[E])==0&&(O=false));break;case 5:for(F=j(v(Q.S7,1),H,6,15,[10,5,18,9,20]),O=true,it=0;it<5;it++)if((g&F[it])==F[it])switch(n.Tl(t.i,lt[it])){case 6:n.Bl(t.i,lt[it])==-1&&(P[A]=true,D[A]=it,(l&F[it])==0&&(O=false));break;case 7:n.Bl(t.i,lt[it])<=0&&(P[A]=true,D[A]=it);break;case 8:P[A]=true,D[A]=it;break;case 16:case 34:case 52:n.$o(t.i,lt[it])==2&&(P[A]=true,D[A]=it);}break;case 6:O=true,(g&21)==21&&(P[A]=true,(l&21)==0&&(O=false)),(g&42)==42&&(P[A]=true,(l&42)==0&&(O=false));break;case 7:for(S=j(v(Q.S7,1),H,6,15,[42,21,74,37,82,41,84]),O=true,R=0;R<7;R++)if((g&S[R])==S[R]){if(n.Tl(t.i,lt[R])==6&&(n.Bl(t.i,lt[R])==1||u&&n.mt(t,lt[R]))||n.Tl(t.i,lt[R])==5&&n.Bl(t.i,lt[R])==0)P[A]=true,D[A]=R,(l&S[R])==0&&(O=false);else for(x=0;x<2;x++)if(bt=(R+6+x)%7,Qt=ut[bt],st=e[A][bt],st!=-1&&!P[st]&&s.xj(t.k,st).length==5){for(wt=s.xj(t.k,st),J=0;wt[J]!=Qt;)++J;for(V=false,K=0,At=1;At<=4;At++){if(b=wt[(J+At)%5],ot=n.fm(t.i,b)>1||n.im(t.i,b)==8||u&&n.wt(t,b),ot&&V){K=0;break;}V=ot,V&&++K;}if(K==2){P[A]=true,P[st]=true,i[A]=true,i[st]=true,s.sj(t.b,h.sjb(Qt));break;}}}}return P[A]&&!O&&(i[A]=true),P[A]?true:!ct;},n.at=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b,g,k;for(i=B(Q.S7,E1,7,t.j.a.length,0,2),l=0;l<t.j.a.length;l++)for(i[l]=B(Q.S7,H,6,s.xj(t.j,l).length,15,1),U=0;U<s.xj(t.j,l).length;U++)i[l][U]=-1;for(g=B(Q.S7,H,6,t.i.g,15,1),c=0;c<t.k.a.length;c++)if(b=s.xj(t.k,c),b.length==3||b.length>=5&&b.length<=7)for(o=0;o<b.length;o++)u=b[o],n.$o(t.i,n.Zl(t.i,0,u))==3&&n.$o(t.i,n.Zl(t.i,1,u))==3&&(g[u]>0?(i[g[u]>>>16][g[u]&32767]=c,i[c][o]=g[u]>>>16):g[u]=(c<<16)+UA+o);for(D=B(Q.xfb,Tt,6,t.j.a.length,16,1),k=0,y=-1;k>y;)for(y=k,T=0;T<t.j.a.length;T++)D[T]||n._s(t,T,i,D,A,e,r,P)&&(D[T]=true,++k);},n.bt=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b;for(P=n.Zl(t.i,0,A),i=n.Zl(t.i,1,A),o=B(Q.S7,H,6,t.i.f,15,1),l=B(Q.S7,H,6,t.i.f,15,1),U=B(Q.S7,H,6,t.i.f,15,1),o[0]=P,o[1]=i,l[P]=1,l[i]=2,U[P]=-1,U[i]=P,u=1,y=1;u<=y;){for(T=0;T<n.$o(t.i,o[u]);T++){if(D=n.Zo(t.i,o[u],T),u>1&&D==P){for(b=B(Q.S7,H,6,l[o[u]],15,1),r=o[u],c=0;c<b.length;c++)b[c]=r,r=U[r];return b;}l[D]==0&&!e[D]&&(o[++y]=D,l[D]=l[o[u]]+1,U[D]=o[u]);}++u;}return null;},n.ct=function(t,A,e){var r,P;for(P=s.xj(t.j,A),r=0;r<P.length;r++)if(e==P[r])return r;return  -1;},n.dt=function(t,A){return t.a[A]&n1;},n.et=function(t,A,e){var r,P;for(P=s.xj(t.k,A),r=0;r<P.length;r++)if(e==P[r])return r;return  -1;},n.ft=function(t,A){return t.c[A]&n1;},n.gt=function(t,A){return t.d[A];},n.ht=function(t,A){return s.xj(t.j,A);},n.it=function(t,A){return s.xj(t.k,A);},n.jt=function(t,A){var e,r,P,i,D;for(i=A.length,D=B(Q.S7,H,6,i,15,1),r=0;r<i;r++)for(e=r==i-1?A[0]:A[r+1],P=0;P<n.$o(t.i,A[r]);P++)if(n.Zo(t.i,A[r],P)==e){D[r]=n._o(t.i,A[r],P);break;}return D;},n.kt=function(t,A){return s.xj(t.k,A).length;},n.lt=function(t,A,e){var r;for(r=0;r<t.k.a.length;r++)if(n.rt(t,r,A)&&n.rt(t,r,e))return r;return  -1;},n.mt=function(t,A){var e;for(e=0;e<n.$o(t.i,A);e++)if(n.ap(t.i,A,e)==2&&n.Tl(t.i,n.Zo(t.i,A,e))==8)return  true;return  false;},n.nt=function(t,A){return t.e[A];},n.ot=function(t,A){return (t.a[A]&H1)!=0;},n.pt=function(t,A){return (t.c[A]&H1)!=0;},n.qt=function(t,A,e){var r,P,i,D;for(P=s.xj(t.j,A),i=0,D=P.length;i<D;++i)if(r=P[i],e==r)return  true;return  false;},n.rt=function(t,A,e){var r,P,i,D;for(P=s.xj(t.k,A),i=0,D=P.length;i<D;++i)if(r=P[i],e==r)return  true;return  false;},n.st=function(t,A){return t.f[A];},n.tt=function(t,A){return (t.a[A]&r0)!=0;},n.ut=function(t,A){return (t.c[A]&r0)!=0;},n.vt=function(t,A){return (t.a[A]&yA)!=0;},n.wt=function(t,A){var e,r,P,i,D,u;for(D=0;D<2;D++)if(e=n.Zl(t.i,D,A),n.Tl(t.i,e)==7&&n.$o(t.i,e)==2&&(r=n.Zl(t.i,1-D,A),n.Tl(t.i,r)==6)){for(u=0;u<n.$o(t.i,r);u++)if(P=n.Zo(t.i,r,u),i=n._o(t.i,r,u),(n.Tl(t.i,P)==8||n.Tl(t.i,P)==16)&&n.fm(t.i,i)==2&&n.$o(t.i,P)==1)return  true;}return  false;},n.xt=function(t,A){var e,r,P,i,D;if(t.i.K){if(Rt(z(n.Ol(t.i,A),1),0))return  true;if(D=n.Jl(t.i,A),D!=null){for(r=D,P=0,i=r.length;P<i;++P)if(e=r[P],e==5||e==6||e==7||e==8||e==15||e==16||e==33||e==34)return  true;return  false;}}return n.Ft(n.Tl(t.i,A));},n.yt=function(t,A){var e,r,P,i,D;if(t.i.K){if(Rt(z(n.Ol(t.i,A),1),0))return  false;if(e=n.Jl(t.i,A),e!=null){for(P=e,i=0,D=P.length;i<D;++i)if(r=P[i],!n.qo(r))return  false;return  true;}}return n.qo(n.Tl(t.i,A));},n.zt=function(t,A){return n.fm(t.i,A)>1||n.im(t.i,A)==8;},n.At=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F;for(F=0;F<t.e.length;F++)if(t.e[F]){for(k=false,r=s.xj(t.j,F),i=0,u=r.length;i<u;++i)A=r[i],t.a[A]|=H1,n.yt(t,A)&&(k=true);for(U=s.xj(t.k,F),T=0,g=U.length;T<g;++T)o=U[T],t.c[o]|=H1;if(t.f[F]){for(e=s.xj(t.j,F),P=0,D=e.length;P<D;++P)A=e[P],t.a[A]|=r0;for(l=s.xj(t.k,F),y=0,b=l.length;y<b;++y)o=l[y],t.c[o]|=r0;}if(k){for(e=s.xj(t.j,F),P=0,D=e.length;P<D;++P)A=e[P],t.a[A]|=yA;for(l=s.xj(t.k,F),y=0,b=l.length;y<b;++y)o=l[y],t.c[o]|=yA;}}for(c=new s.trb(t.b);c.a<c.c.a.length;)o=s.srb(c).a,t.c[o]=0;},n.Bt=function(t,A,e){var r,P,i,D,u,o,l,U,y,T;for(T=A.length,P=A,i=0,D=P.length;i<D;++i)r=P[i],y=t.a[r]&n1,(y==0||y>T)&&(t.a[r]&=fA,t.a[r]|=T);for(o=e,l=0,U=o.length;l<U;++l)u=o[l],y=t.c[u]&n1,(y==0||y>T)&&(t.c[u]&=fA,t.c[u]|=T);},n.Ct=function(t,A,e){var r;for(r=s.xj(t.k,A).length;e>=r;)e-=r;for(;e<0;)e+=r;return e;},n.Dt=function(t,A){n.Et.call(this,t,A);},n.Et=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S;this.i=t,this.g=7,this.j=new s.Mj(),this.k=new s.Mj(),this.b=new s.Mj(),this.a=B(Q.S7,H,6,this.i.f,15,1),this.c=B(Q.S7,H,6,this.i.g,15,1),this.i.$b(1),T=B(Q.xfb,Tt,6,this.i.f,16,1),c=B(Q.xfb,Tt,6,this.i.g,16,1);do for(D=false,e=0;e<this.i.f;e++)if(!T[e]){for(k=0,y=0;y<n.$o(this.i,e);y++)T[n.Zo(this.i,e,y)]||++k;if(k<2){for(T[e]=true,U=0;U<n.$o(this.i,e);U++)c[n._o(this.i,e,U)]=true;D=true;}}while(D);for(S=0;S<this.i.f&&T[S];)++S;if(S!=this.i.f){for(o=B(Q.S7,H,6,this.i.f,15,1),o[0]=S,g=B(Q.S7,H,6,this.i.f,15,1),g[0]=-1,u=B(Q.S7,H,6,this.i.f,15,1),u[S]=1,i=0,l=0,b=1;i<=l;){for(U=0;U<n.$o(this.i,o[i]);U++)if(P=n.Zo(this.i,o[i],U),P!=g[o[i]]){if(u[P]!=0){n.$s(this,n._o(this.i,o[i],U),T);continue;}T[P]||(u[P]=b,g[P]=o[i],o[++l]=P);}if(++i,i>l){for(e=0;e<this.i.f;e++)if(u[e]==0&&!T[e]){u[e]=++b,o[++l]=e,g[e]=-1;break;}}}if((A&4)!=0&&(this.e=B(Q.xfb,Tt,6,this.j.a.length,16,1),this.f=B(Q.xfb,Tt,6,this.j.a.length,16,1),this.d=B(Q.S7,H,6,this.j.a.length,15,1),n.at(this,this.e,this.f,this.d,(A&8)!=0),n.At(this)),(A&2)!=0)for(r=0;r<this.i.g;r++)!c[r]&&n.fm(this.i,r)!=0&&(F=n.bt(this,r,T),F!=null&&n.Bt(this,F,n.jt(this,F)));}},n.Ft=function(t){return t==5||t==6||t==7||t==8||t==15||t==16||t==33||t==34;},I(151,1,{},n.Dt),L.g=0,Q.E8=N(151),n.Gt=function(t,A){var e,r,P,i,D,u,o,l,U,y,T;if(A==2||A==5)s.sj(t.F,n.ku(t.G,t.G.length)),t.b&&s.sj(t.a,n.lu(t.C,t.C.length));else if(A==4)T=n.St(n.ku(t.G,t.G.length)),s.Mxb(t.O,T)||(s.Kxb(t.O,T),s.sj(t.F,n.ku(t.G,t.G.length)),t.b&&s.sj(t.a,n.lu(t.C,t.C.length)));else if(A==3){if(T=n.St(n.ku(t.G,t.G.length)),!s.Mxb(t.O,T)){for(y=false,l=(u=new s.hxb(new s.nxb(new s.uqb(t.O.a).a).b),new s.Dqb(u));s.cpb(l.a.a);){for(o=(D=s.fxb(l.a),D.hd()),U=0,r=T,P=0,i=r.length;P<i;++P){for(e=r[P];U<o.length&&o[U]<e;)++U;if(U<o.length&&e==o[U]){y=true;break;}}if(y)break;}y||(s.Kxb(t.O,T),s.sj(t.F,n.ku(t.G,t.G.length)),t.b&&s.sj(t.a,n.lu(t.C,t.C.length)));}}else A==6&&(T=n.Tt(t,n.ku(t.G,t.G.length)),s.Mxb(t.O,T)||(s.Kxb(t.O,T),s.sj(t.F,n.ku(t.G,t.G.length)),t.b&&s.sj(t.a,n.lu(t.C,t.C.length))));},n.Ht=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c;if(u=n.$o(t.H,A),r=t.o[e],r>u)return  false;if(l=n.Ol(t.H,A),i=n.Ol(t.g,e),P=n.Jl(t.g,e),o=n.Jl(t.H,A),Rt(z(i,1),0)){if(P!=null){if(Rt(z(l,1),0)){if(o==null||!n.Yt(P,o))return  false;}else if(o!=null){if(n.$t(o,P))return  false;}else if(n.Xt(n.Tl(t.H,A),P))return  false;}}else {if(Rt(z(l,1),0))return  false;if(P!=null){if(o!=null){if(!n.Yt(o,P))return  false;}else if(!n.Xt(n.Tl(t.H,A),P))return  false;}else if(o!=null||t.J[A]!=t.k[e])return  false;}if(Rt(Pt(l,i),0)){if(Rt(z(i,B1),0)){if(t.H.K&&S1(z(l,B1),0))return  false;if(r!=u)return  false;}if(Rt(z(i,Z1),0)&&r>=u&&S1(z(l,Z1),0))return  false;}if(Rt(z(t.I[A],tA(t.j[e])),0)||Rt(z(t.B[e],tA(t.M[A])),0))return  false;if(D=z(i,EA),t.H.K){if(U=z(i,EA),nt(U,0)!=0&&(nt(D,0)==0||Rt(z(D,tA(U)),0)))return  false;}else if(nt(D,0)!=0&&S1(z(D,t.M[A]),0))return  false;return n.Bl(t.g,e)!=0&&n.Bl(t.g,e)!=n.Bl(t.H,A)||n.Ml(t.g,e)!=0&&n.Ml(t.g,e)!=n.Ml(t.H,A)||n.Pl(t.g,e)!=0&&n.Pl(t.g,e)!=n.Pl(t.H,A)||(y=z(i,uP),nt(y,0)!=0&&n.xp(t.H,A)!=zt(A0(y,47))-7)?false:(T=z(i,FA),nt(T,0)!=0?(T=A0(T,22),t.H.K?(c=zt(A0(z(n.Ol(t.H,A),FA),22)),zt(T)==c):n.Uo(t.H,A)==zt(T)):true);},n.It=function(t,A,e){var r,P,i,D,u,o,l,U,y,T;if(l=t.K[A],D=t.n[e],(n.hm(t.g,e)&Ln)!=0){if(o=n.jm(t.H,A),P=n.jm(t.g,e),i=n.hm(t.g,e)&127,o!=P&&!(o==1&&(i&1)!=0)&&!(o==2&&(i&2)!=0)&&!(o==4&&(i&4)!=0)&&!(o==32&&(i&32)!=0)&&!(o==64&&(i&64)!=0)&&!(o==16&&(i&16)!=0)&&!(o==8&&(i&8)!=0))return  false;l&=-128,D&=-128;}if((l&~D)!=0)return  false;if(T=(n.hm(t.g,e)&iA)>>17,T!=0){if(t.H.K&&T==(n.hm(t.H,e)&iA)>>17)return  true;if(T<=2)return U=n.Yo(t.H,A),T==1?U>=8&&U<=12:U>=12;for(r=false,y=n.Ep(t.H),u=0;u<y.j.a.length;u++)if(s.xj(y.k,u).length==T&&n.rt(y,u,A)){r=true;break;}if(!r)return  false;}return  true;},n.Jt=function(t){var A,e;for(t.b=null,A=0;A<t.g.g;A++)n.Cm(t.g,A)&&(!t.b&&(t.b=new s.Mj()),e=new n.mu(),e.a=n.Zl(t.g,0,A),e.b=n.Zl(t.g,1,A),e.d=n._l(t.g,A),e.c=n.$l(t.g,A),s.sj(t.b,e));},n.Kt=function(t){var A,e,r,P,i,D,u,o,l,U,y,T;for(n.bw(t.g,t.N),o=d.Math.max(t.g.f,t.g.g)+16,t.s=B(Q.S7,H,6,o,15,1),t.u=B(Q.S7,H,6,o,15,1),t.v=B(Q.S7,H,6,o,15,1),t.t=B(Q.xfb,Tt,6,o+1,16,1),D=B(Q.xfb,Tt,6,t.g.f,16,1),u=B(Q.xfb,Tt,6,t.g.g,16,1),r=0,e=0;e<t.g.f;e++)if(!t.D[e]&&!D[e])for(t.s[r]=e,t.v[r]=-1,t.u[r]=-1,l=r;r<=l;){for(U=0;U<n.Mo(t.g,t.s[r]);U++)l=n.hu(t,r,l,U,D,u,-1);for(;t.t[++r];);}if(t.w=r,t.p!=0){for(l=t.w-1,i=0;i<t.d;i++)for(r=0;r<=l;){for(y=0;y<n.Mo(t.g,t.s[r]);y++)l=n.hu(t,r,l,y,D,u,i);for(;t.t[++r];);}for(A=0;A<t.g.f;A++)if(t.D[A]&&!D[A])for(t.s[r]=A,t.v[r]=-1,t.u[r]=-1,l=r;r<=l;){for(y=0;y<n.Mo(t.g,t.s[r]);y++)n.Zo(t.g,t.s[r],y)<t.g.f&&(l=n.hu(t,r,l,y,D,u,t.f[A]));for(;t.t[++r];);}for(t.e=B(Q.S7,H,6,t.d,15,1),T=0;T<t.d;T++)t.e[T]=-1;for(U=t.w;U<r;U++)P=t.f[t.s[U]],t.e[P]==-1&&(t.e[P]=U);}t.A=r;},n.Lt=function(t,A){var e,r,P,i,D,u,o,l;for(l=n.Ep(t.H),u=0;u<l.j.a.length;u++)if(n.rt(l,u,A)){for(o=B(Q.S7,H,6,2,15,1),P=0;P<2;P++)for(o[P]=Ot,r=n.Zl(t.H,P,A),i=0;i<n.$o(t.H,r);i++)e=n.Zo(t.H,r,i),e!=n.Zl(t.H,1-P,A)&&o[P]>e&&(o[P]=e);return D=0,n.qt(l,u,o[0])&&++D,n.qt(l,u,o[1])&&++D,D==2?2:D==1?1:2;}return 0;},n.Mt=function(t,A,e){var r,P,i,D,u;if(t.b){for(t.C=B(Q.xfb,Tt,6,A.length,16,1),P=new s.trb(t.b);P.a<P.c.a.length;)if(r=s.srb(P),t.f==null||e==-1&&t.f[r.a]==-1&&t.f[r.b]==-1||e!=-1&&(t.f[r.a]==e||t.f[r.b]==e)){if(u=B(Q.S7,H,6,r.c+2,15,1),i=n.yp(t.H,u,t.G[r.a],t.G[r.b],r.c+1,A,null)-1,i<r.d||i>r.c)return  false;for(D=1;D<=i;D++)t.C[u[D]]=true;}}return  true;},n.Nt=function(t,A){var e,r,P,i,D,u,o,l;for(P=0;P<t.g.g;P++)if((n.hm(t.g,P)&TA)!=0){if(i=n.gm(t.g,P),i==0)continue;if(e=n.Zl(t.g,0,P),r=n.Zl(t.g,1,P),t.f==null||A==-1&&t.f[e]==-1&&t.f[r]==-1||A!=-1&&(t.f[e]==A||t.f[r]==A)){if(D=t.G[e],u=t.G[r],o=n.Xo(t.H,D,u),l=n.gm(t.H,o),l==0&&(n.jq(t.H,o)&&(l=n.Lt(t,o)),l==0)||i==3||l==3)continue;if(n.Ut(t,P)==(i==l))return  false;}}return  true;},n.Ot=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g;for(D=0,o=0;o<t.g.f;o++)if((t.f==null||t.f[o]==A)&&Rt(z(n.Ol(t.g,o),U0),0)){if(T=t.G[o],y=n.Nl(t.g,o),b=n.Nl(t.H,T),y==0||y==3)continue;if(b==0||b==3)return  false;if(n.Hl(t.g,o)==1){++D;continue;}if(n.Hl(t.H,T)==1)return  false;if(n.Hl(t.g,o)==2){++D;continue;}if(n.Hl(t.H,T)==2||n.Zt(t,o)==(y==b))return  false;}if(D!=0){for(P=B(Q.S7,H,6,D,15,1),i=0,l=0;l<t.g.f;l++)(t.f==null||t.f[l]==A)&&Rt(z(n.Ol(t.g,l),U0),0)&&(y=n.Nl(t.g,l),y!=0&&y!=3&&(P[i++]=n.Gl(t.g,l)<<24|n.Hl(t.g,l)<<22|l));for(w.nAb(P,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),i=0;i<P.length;)for(U=P[i]&W1,c=t.G[U],e=P[i]&-4194304,r=n.Zt(t,U)^n.Nl(t.g,U)==n.Nl(t.H,c),++i;i<P.length&&(P[i]&-4194304)==e;i++)if(u=P[i]&W1,T=t.G[u],n.Hl(t.H,T)!=n.Hl(t.H,c)||n.Gl(t.H,T)!=n.Gl(t.H,c)||(g=n.Zt(t,u)^n.Nl(t.g,u)==n.Nl(t.H,T),g!=r))return  false;}return  true;},n.Pt=function(t,A,e){var r,P,i,D,u,o,l;if(t.P=false,t.F=new s.Mj(),t.a=new s.Mj(),s.Kwb(t.O.a),!t.H||!t.g||t.g.f-t.p>t.H.f||t.g.g-t.q>t.H.g||t.g.f==0)return 0;for(A==6&&(t.N=63),n.du(t,e),r=B(Q.xfb,Tt,6,t.H.f,16,1),t.G=B(Q.S7,H,6,t.g.f,15,1),s.Frb(t.G,-1),u=B(Q.S7,H,6,t.A,15,1),s.Lrb(u,u.length,-1),i=0;!t.P;){if(t.w!=0){if(l=t.u[i]==-1?t.H.f:n.Mo(t.H,t.G[t.u[i]]),++u[i],u[i]==l){if(u[i]=-1,i==0)break;--i,t.t[i]||(r[t.G[t.s[i]]]=false);continue;}if(t.u[i]==-1)r[u[i]]||n.Ht(t,u[i],t.s[i])&&(t.G[t.s[i]]=u[i],r[u[i]]=true,++i);else {if(n.Zo(t.H,t.G[t.u[i]],u[i])>=t.H.f)continue;P=n.Zo(t.H,t.G[t.u[i]],u[i]),t.t[i]?P==t.G[t.s[i]]&&n.It(t,n._o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&++i:r[P]||n.Ht(t,P,t.s[i])&&n.It(t,n._o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&(r[P]=true,t.G[t.s[i]]=P,++i);}}if(i==t.w){if(n.Ot(t,-1)&&n.Nt(t,-1)&&n.Mt(t,r,-1)){for(o=false,D=0;D<t.d;D++)if(n.Vt(t,r,u,D)){o=true;break;}if(A==1&&!o||!o&&(n.Gt(t,A),A==2))return 1;}if(i==0)break;--i,t.t[i]||(r[t.G[t.s[i]]]=false);}}return t.F.a.length;},n.Qt=function(t,A){var e,r,P,i,D,u,o,l;if(o=0,t.K){switch(A<t.f&&n.vt(t.p,A)?o=Pt(o,LP):A<t.f&&n.ot(t.p,A)&&(o=Pt(o,2)),l=(P=t.u[A]&x0,P==0?0:P==f0?2:P==B1?3:4),l!=0&&(o=Pt(o,8),l>2&&(o=Pt(o,16)),l>3&&(o=Pt(o,32))),e=t.s[A],e<0?o=Pt(o,Qn):e>0&&(o=Pt(o,on)),D=t.j[A],D){case 0:break;case 1:o=Pt(o,r0);break;case 2:o=Pt(o,tn);break;case 3:o=Pt(o,iA);break;default:o=Pt(o,ln);}switch(r=n.Po(t,A),r){case 0:break;case 1:o=Pt(o,Mn);break;case 2:o=Pt(o,Ke);break;case 3:o=Pt(o,ze);break;default:o=Pt(o,De);}u=t.o[A],u>0&&(o=Pt(o,N0)),u>1&&(o=Pt(o,UA));}else {switch(A<t.f&&n.vt(t.p,A)?o=Pt(o,LP):A<t.f&&n.ot(t.p,A)?o=Pt(o,2):o=Pt(o,4),(t.u[A]&O0)!=0?o=Pt(o,Ae):o=Pt(o,ne),l=(P=t.u[A]&x0,P==0?0:P==f0?2:P==B1?3:4),l==0?o=Pt(o,112):l==2?o=Pt(o,104):l==3?o=Pt(o,88):o=Pt(o,56),e=t.s[A],e==0?o=Pt(o,re):e<0?o=Pt(o,Qn):o=Pt(o,on),i=t.e[A]-n.tp(t,A)+n.op(t,A),i){case 0:o=Pt(o,1792);break;case 1:o=Pt(o,1664);break;case 2:o=Pt(o,1408);break;default:o=Pt(o,896);}switch(D=t.j[A],D){case 0:o=Pt(o,3932160);break;case 1:o=Pt(o,Pe);break;case 2:o=Pt(o,pn);break;case 3:o=Pt(o,Cn);break;default:o=Pt(o,ln);}switch(r=n.Po(t,A),r){case 0:o=Pt(o,Ve);break;case 1:o=Pt(o,Ie);break;case 2:o=Pt(o,Ne);break;case 3:o=Pt(o,Ge);break;default:o=Pt(o,De);}switch(u=t.o[A],u){case 0:o=Pt(o,98304);break;case 1:o=Pt(o,81920);break;default:o=Pt(o,49152);}}return o;},n.Rt=function(t,A){var e;if(e=0,A<t.g&&(n.ut(t.p,A)||t.H[A]==8)||t.H[A]==8)e|=8;else switch(n.fm(t,A)){case 0:e|=16;break;case 1:e|=1;break;case 2:e|=2;break;case 3:e|=4;break;case 4:e|=32;break;case 5:e|=64;}return (t.F[A]&64)!=0?e|=256:t.K||(e|=128),A<t.g&&n.pt(t.p,A)?e|=j0:t.K||(e|=_0),e;},n.St=function(t){var A,e,r,P,i,D,u,o,l,U;for(o=0,r=t,i=0,u=r.length;i<u;++i)A=r[i],A==-1&&++o;if(o!=0)for(U=t,t=B(Q.S7,H,6,U.length-o,15,1),l=0,e=U,P=0,D=e.length;P<D;++P)A=e[P],A!=-1&&(t[l++]=A);return w.nAb(t,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),t;},n.Tt=function(t,A){var e,r,P,i,D,u,o,l;for(D=0,r=A,P=0,i=r.length;P<i;++P)e=r[P],e==-1&&++D;for(l=B(Q.S7,H,6,A.length-D,15,1),o=0,u=0;u<A.length;u++)A[u]!=-1&&(l[o]=n.mw(t.g,u)<<16|n.mw(t.H,A[u]),t.i!=null&&(l[o]|=t.i[u]<<24),++o);return w.nAb(l,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[]))),l;},n.Ut=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c;for(i=false,P=0;P<2;P++)if(e=n.Zl(t.g,P,A),U=t.G[e],n.$o(t.H,U)>2){for(T=n.Zl(t.g,1-P,A),o=Ot,u=0;u<n.$o(t.g,e);u++)r=n.Zo(t.g,e,u),r!=T&&o>r&&(o=r);for(c=t.G[T],l=Ot,D=0;D<n.$o(t.H,U);D++)y=n.Zo(t.H,U,D),y!=c&&l>y&&(l=y);t.G[o]!=l&&(i=!i);}return i;},n.Vt=function(t,A,e,r){var P,i,D,u,o,l,U,y;for(u=t.e[r],o=u+1;o<t.A&&t.f[t.s[o]]==r;)++o;for(U=u;U<o;U++)e[U]=-1;for(D=u;;){if(y=t.u[D]==-1?t.H.f:n.Mo(t.H,t.G[t.u[D]]),++e[D],e[D]==y){if(e[D]=-1,D==u)break;--D,t.t[D]||(A[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);continue;}if(t.u[D]==-1)A[e[D]]||n.Ht(t,e[D],t.s[D])&&(t.G[t.s[D]]=e[D],A[e[D]]=true,++D);else {if(n.Zo(t.H,t.G[t.u[D]],e[D])>=t.H.f)continue;i=n.Zo(t.H,t.G[t.u[D]],e[D]),t.t[D]?i==t.G[t.s[D]]&&n.It(t,n._o(t.H,t.G[t.u[D]],e[D]),t.v[D])&&++D:A[i]||n.Ht(t,i,t.s[D])&&n.It(t,n._o(t.H,t.G[t.u[D]],e[D]),t.v[D])&&(A[i]=true,t.G[t.s[D]]=i,++D);}if(D==o){if(n.Ot(t,r)&&n.Nt(t,r)&&n.Mt(t,A,r)){for(l=u;l<o;l++)t.t[l]||(P=t.s[l],A[t.G[P]]=false,t.G[P]=-1);return  true;}--D,t.t[D]||(A[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);}}return  false;},n.Wt=function(t){return n.Pt(t,1,t.c)>0;},n.Xt=function(t,A){var e;for(e=0;e<A.length;e++)if(A[e]==t)return  true;return  false;},n.Yt=function(t,A){var e,r,P;for(P=0,r=0;r<t.length;r++){for(e=t[r];A[P]<e;)if(++P,P==A.length)return  false;if(A[P]>e)return  false;}return  true;},n.Zt=function(t,A){var e,r,P,i,D,u,o,l;if(D=false,n.Qo(t.g,A)==0)for(i=1;i<n.$o(t.g,A);i++)for(u=0;u<i;u++)r=n.Zo(t.g,A,i),P=n.Zo(t.g,A,u),t.G[r]>t.G[P]^r>P&&(D=!D);else for(i=0;i<n.$o(t.g,A);i++){for(e=n.Zo(t.g,A,i),l=0,o=B(Q.S7,H,6,3,15,1),u=0;u<n.$o(t.g,e);u++)o[l]=n.Zo(t.g,e,u),o[l]!=A&&++l;l==2&&t.G[o[0]]>t.G[o[1]]^o[0]>o[1]&&(D=!D);}return D;},n.$t=function(t,A){var e,r,P,i;for(P=0,i=0;P<t.length&&i<A.length;){if(e=t[P],r=A[i],e==r)return  true;e<r?++P:++i;}return  false;},n._t=function(t,A){var e,r,P,i,D;if(!A||A.q==0||!A.K){t.g=null;return;}for(t.g=A,t.r=false,n.bw(t.g,1),t.N=7,r=0;r<t.g.f;r++)Rt(z(n.Ol(t.g,r),{l:U0,m:0,h:3}),0)&&(t.N=15);for(D=0;D<t.g.g;D++)(n.hm(t.g,D)&TA)!=0&&(t.N=15);for(t.L&&t.N!=7&&n.bw(t.H,t.N),t.p=0,t.q=0,t.D=B(Q.xfb,Tt,6,t.g.f,16,1),P=0;P<t.g.f;P++)t.D[P]=n.Jm(t.g,P),t.D[P]&&++t.p;if(t.d=0,t.f=null,t.i=null,t.p!=0){for(i=0;i<t.g.g;i++)(t.D[n.Zl(t.g,0,i)]||t.D[n.Zl(t.g,1,i)])&&++t.q;for(e=0;e<t.g.q;e++)n.sn(t.g,e,t.D[e]);t.f=B(Q.S7,H,6,t.g.q,15,1),t.d=n.hp(t.g,t.f,true,false);}},n.au=function(t,A){t.i=A;},n.bu=function(t,A,e){n.cu(t,e),n._t(t,A);},n.cu=function(t,A){if(!A||A.q==0){t.H=null;return;}t.H=A,t.L=false,n.bw(t.H,1);},n.du=function(t,A){t.L||(n.gu(t,A),t.L=true),t.r||(n.eu(t,A),n.Kt(t),n.Jt(t),t.r=true);},n.eu=function(t,A){var e,r,P,i,D,u,o,l,U,y;for(i=null,o=null,D=null,n.bw(t.g,t.N),t.o=B(Q.S7,H,6,t.g.f,15,1),r=0;r<t.g.f;r++)t.o[r]=n.$o(t.g,r);if(t.p!=0){for(l=new n.uw(t.g.q,t.g.r),y=B(Q.xfb,Tt,6,t.g.q,16,1),P=0;P<t.g.q;P++)y[P]=!t.D[P];for(n.Co(t.g,l,y,true,null),n.bw(l,t.N),n.fu(t,l,A),i=t.j,o=t.n,D=t.k,U=0,e=0;e<t.g.f;e++)t.D[e]||(t.o[e]=n.$o(l,U++));}if(n.fu(t,t.g,A),t.p!=0){for(U=0,e=0;e<t.g.q;e++)t.D[e]||(t.j[e]=i[U],t.k[e]=D[U++]);for(U=0,u=0;u<t.g.r;u++)!t.D[n.Zl(t.g,0,u)]&&!t.D[n.Zl(t.g,1,u)]&&(t.n[u]=o[U++]);}},n.fu=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S;for(g=A.f,t.j=B(Q.T7,e0,6,A.f,14,1),t.k=B(Q.S7,H,6,A.f,15,1),u=0;u<g;u++)t.j[u]=en(z(Pt(n.Qt(A,u),A.B[u]),sP),W8),t.k[u]=A.C[u],(e&1)!=0&&(t.k[u]+=A.s[u]+16<<8),(e&2)!=0&&(t.k[u]+=A.A[u]<<16);for(t.B=B(Q.T7,e0,6,A.f,14,1),F=(n.bw(A,7),A.p),b=0;b<F.j.a.length;b++){for(c=false,l=s.xj(F.k,b),U=0,y=l.length;U<y;++U)if(T=l[U],(A.G[T]&LA)!=0){c=true;break;}if(!c)for(S=s.xj(F.k,b).length,P=s.xj(F.j,b),i=0,D=P.length;i<D;++i)r=P[i],S==3?t.B[r]=Pt(t.B[r],OA):S==4?t.B[r]=Pt(t.B[r],xA):S==5?t.B[r]=Pt(t.B[r],qA):S==6?t.B[r]=Pt(t.B[r],VA):S==7&&(t.B[r]=Pt(t.B[r],IA));}for(k=A.g,t.n=B(Q.S7,H,6,A.g,15,1),o=0;o<k;o++)t.n[o]=(n.Rt(A,o)|A.G[o])&6291967^6291840,(e&4)!=0?(t.n[o]&2)!=0&&(t.n[o]|=8):(e&8)!=0&&(t.n[o]&2)!=0&&o<A.g&&n.pt(A.p,o)&&(t.n[o]|=8);},n.gu=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c;for(n.bw(t.H,t.N),U=t.H.f,t.J=B(Q.S7,H,6,U,15,1),t.I=B(Q.T7,e0,6,U,14,1),D=0;D<U;D++)t.I[D]=en(z(Pt(n.Qt(t.H,D),n.Ol(t.H,D)),sP),W8),t.J[D]=n.Tl(t.H,D),(A&1)!=0&&(t.J[D]+=n.Bl(t.H,D)+16<<8),(A&2)!=0&&(t.J[D]+=n.Ml(t.H,D)<<16);for(t.M=B(Q.T7,e0,6,U,14,1),T=n.Ep(t.H),l=0;l<T.j.a.length;l++)for(c=s.xj(T.k,l).length,r=s.xj(T.j,l),P=0,i=r.length;P<i;++P)u=r[P],c==3?t.M[u]=Pt(t.M[u],OA):c==4?t.M[u]=Pt(t.M[u],xA):c==5?t.M[u]=Pt(t.M[u],qA):c==6?t.M[u]=Pt(t.M[u],VA):c==7&&(t.M[u]=Pt(t.M[u],IA));for(e=0;e<U;e++)c=n.Uo(t.H,e),c==0?t.M[e]=Pt(t.M[e],lA):c>7&&(t.M[e]=Pt(t.M[e],un));for(y=t.H.g,t.K=B(Q.S7,H,6,y,15,1),o=0;o<y;o++)t.K[o]=(n.Rt(t.H,o)|n.hm(t.H,o))&6422527^6291840;},n.hu=function(t,A,e,r,P,i,D){var u,o;return u=n.Zo(t.g,t.s[A],r),u<t.D.length&&(!t.D[u]||t.f[u]==D)&&u!=t.u[A]&&(o=n._o(t.g,t.s[A],r),!i[o]&&!n.Cm(t.g,o)&&(t.s[++e]=u,t.u[e]=t.s[A],t.v[e]=o,i[o]=true,P[u]?t.t[e]=true:P[u]=true)),e;},n.iu=function(){this.c=8,this.O=new s.Qxb(new Ut.I4());},n.ju=function(t){this.c=t,this.O=new s.Qxb(new Ut.I4());},n.ku=function(t,A){var e;return e=B(Q.S7,H,6,A,15,1),h.Skb(t,0,e,0,d.Math.min(t.length,A)),e;},n.lu=function(t,A){var e;return e=B(Q.xfb,Tt,6,A,16,1),h.Skb(t,0,e,0,d.Math.min(t.length,A)),e;},I(64,1,{},n.iu,n.ju),L.c=0,L.d=0,L.p=0,L.q=0,L.r=false,L.w=0,L.A=0,L.L=false,L.N=0,L.P=false,Q.H8=N(64),n.mu=function(){},I(215,1,{215:1},n.mu),L.a=0,L.b=0,L.c=0,L.d=0,Q.F8=N(215),n.pu=function(){n.pu=i1,n.nu=j(v(Q.Qcb,1),yt,2,6,["QM@HzAmdqjF@","RF@Q``","qC`@ISTAlQE`","`J@H","QM@HzAmdqbF@","qC`@ISTAlQEhqPp@","sJP@DiZhAmQEb","RF@QPvR@","QM@HzA@","qC`@ISTAlQEhpPp@","qC`@Qz`MbHl","sJP@DiZhAmQEcFZF@","RFPDXH","qC`@IVtAlQE`","QM@HvAmdqfF@","sGP@DiVj`FsDVM@","`L@H","sJP@DizhAmQEcFBF@","sJP@DjvhAmQEb","sFp@DiTt@@AlqEcP","sGP@LdbMU@MfHlZ","QMHAIhD","QM@HzAy@","sJP@DkVhAmQEb","sNp@DiUjj@[\\QXu`","sJP@DiZhAmQEcFBF@","sGP@DjVj`FsDVM@","RFPDTH","RG@DXOH@","sGP@Divj`FsDVMcAC@","sGP@Dj}j`FsDVM@","qC`@Qz`MbHmFRF@","sNp@LdbJjj@[\\QXu`","QMHAIhGe@","QM@HzAyd`","QM`AIhD","qC`@ISTA@","sGP@DkUj`FsDVM@","qC`@IVtAlQEhqPp@","sNp@DiUjj@[\\QXuqea`@","KAx@@IRjuUPAlHPfES\\","QM`BN`P","sJP@DjZhAmQEcFJF@","Hid@@DjU^nBBH@FtaBXUMp`","sNp@Diujj@[\\QXuq`a`@","sJP@DjvhAmQEcFZF@","sJP@DjZhAmQEcFFF@","sOp@DjWkB@@FwDVM\\YhX@","sNp@Dj}Zj@[\\QXu`","sNp@DiWjj@[\\QXuq`a`@","sOp@DjWkB@@D","KAx@@ITouUPAlHPfES\\","KAx@@YIDTjjh@vDHSBin@","sNp@DkUZj@[\\QXu`","RFPDXOH@","QM`BN`^L`","qC`@ISTAy@","sGP@LdbMU@MfHl[FVF@","qCb@AIZ`H","KAx@@IRjuUPAlHPfES]FFa`@","KAx@@ITnuUPAlHPfES\\","HiD@@DiUVjj`AmHPfES\\H","sNp@DjUjj@[\\QXu`","sJP@DkVhAmQEcFJF@","sGP@DjVj`FsDVMcCC@","qC`@Qz`MbHmFBF@","sJP@DkfhAmQEb","qC`@IVtAlQEhsPp@","sGP@Djuj`FsDVM@","sGP@Dj}j`FsDVMcMC@","sJP@DiZhA@","KAx@@ISjuUPAlHPfES]F@a`@","sJP@DjZhAmQEcFRF@","KAx@@IRnuUPAlHPfES]F@a`@","HiD@@DjWvjj`AmHPfES\\H","QMHAIhGd@","sNp@DiUjj@[\\QXuq`a`@","KAx@@IVjmUPAlHPfES\\","sGP@DjVj`FsDVMcMC@","QM`AIhGe@","HiD@@LdbJRjjh@[RDIaTwB","qCp@AIZ`H","sGP@LdbMU@MfHl[FFF@","QMDARVA@","sNp@LdbJjj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqca`@","sGP@Dkej`FsDVM@","qCb@AIZ`OI@","HaD@@DjUZxHH@AlHPfES]FLa`@","sGP@DkYj`FsDVM@","qCb@AIV`H","sNp@LdbJjj@[\\QXuqea`@","sGP@DkUj`FsDVMcEC@","sFp@DiTt@@Axa@","Hmt@@DjU_ZxHHj@AmhPfES\\Lj","QM`BN`^P","qCb@AIZ`OH`","sFp@DiTt@@AxaP","sGP@Djuj`FsDVMcEC@","sGP@Djuj`FsDVMcIC@","sGP@DkUj`FsDVMcKC@","sJP@DkfhAmQEcFRF@","sGP@DjVj`FsDVMcIC@","HaD@@DjUZxHH@AlHPfES]FFa`@","qC`@IRtDVqDV@","sNp@Dj}Zj@[\\QXuqfa`@","KAx@@ITnuUPAlHPfES]FFa`@","HiD@@DkUUjj`AmHPfES\\H","sJQ@@dkU@H","qC`@Qz`H","KAx@@IUkmUPAlHPfES\\","KAx@@ITouUPAlHPfES]FJa`@","sJP@H~j@[TQX`","sGP@DjZj`FsDVM@","sJP@DkVhAmQEcFFF@","sJX@@eKU@H","sJP@DizhAy@","QMHAIhGbP","KAx@@ITouUPAlHPfES]FNa`@","HaD@@DjUZxHD@AlHPfES\\","HaD@@DjUZxHH@A@","sNp@LdbJjj@[\\QXuqaa`@","Hed@@LdbRQUUUP@vTHSBinFP","KAx@@ITouUPAlHPfES]FLa`@","sNp@DkUZj@[\\QXuqba`@","KAx@@ITjuUPAlHPfES]FNa`@","KAx@@YIDTjjh@vDHSBincGPp@","HaD@@DjYvxH`@AlHPfES]FLa`@","RF@QP`","qCb@AIj`H","sNp@DjUjj@[\\QXuqaa`@","sNp@DkVZj@[\\QXu`","KAx@@YIDUJjh@vDHSBin@","sGP@DkYj`FsDVMcIC@","sGP@DjVj`FsDVMcAC@","sGP@DiVj`D","sJP@DkVhAmQEcFZF@","sNp@LdbLjj@[\\QXu`","QM@HvAmdqbF@","HaD@@DjWjXHB@AlHPfES\\","sNp@DjwZj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqda`@","sFp@DiTt@@Axa`","HiD@@Djuujj`AmHPfES\\H","sNp@DkUZj@[\\QXuqca`@","sJP@DiZhAy@","KAx@@YIDTjjh@vDHSBincCPp@","KAx@@IWNmUPAlHPfES\\","KAx@@IVkMUPAlHPfES\\","sJQ@@dju@H","qCb@AIZ`OH@","qC`@ISTAxa@","sNp@DjyZj@[\\QXu`","Hid@@DjUfaBB`@FtaBXUMp`","HiD@@DiUVjj`AmHPfES\\LXBF@","KAx@@IUjmUPAlHPfES\\","HiD@@DjWvjj`AmHPfES\\LXjF@","sJP@DjVhAmQEb","qCb@AIV`OH`","HiD@@LdbJRjjh@[RDIaTwCFDa`@","KAx@@YIDTjjh@vDHSBinc@Pp@","sNp@DjUjj@[\\QXuqda`@","qC`@Qz`OED","sJP@DkfhAmQEcFZF@","KAx@@YIDbjjh@vDHSBincDPp@","sGP@Djyj`FsDVMcMC@","KAx@@IVrmUPAlHPfES\\","qCp@AIZ`OI@","sJX@@dkU@H","sJQ@@dkU@OH`","sNp@Di]ZjBBvxbqk@","Hkl@@DjU_Uk``bj`@[VDIaTwCJzX","sGP@DjZj`FsDVMcEC@","Hid@@DjU^nBBH@FtaBXUMpqcHX@","sNp@DkeZj@[\\QXu`","sNp@DjYjj@[\\QXuqca`@","sGQ@@djuT@`","HiD@@LdbJTjjh@[RDIaTwB","sOp@DjWkB@@Gd`","HeT@@LdbbRKBDQD@CYPaLJfxY@","qCr@XIKTA@","HiD@@DjW^jj`AmHPfES\\LXJF@","HeT@@DjU]k``b`@[JDIaTwCH","sGP@Djuj`FsDVMcCC@","`IH`B","sOp@DjWkB@@GdX","sJQ@@eKU@H","KAx@@YIDUJjh@vDHSBincBPp@","sJX@@eKU@OH@","KAx@@YIDTjjh@vDHSBincAPp@","sOq@@drm\\@@@`","KAx@@IUkMUPAlHPfES\\","qCp@AIj`H","Hed@@DjUUjjj@FraBXUMpr","sGX@@eJuT@`","sGP@DkUj`FsDVMcCC@","HiD@@Dj}Ujj`AmHPfES\\LXrF@","KAx@@ITouUPAlHPfES]FHa`@","Hed@@DjWujjj@FraBXUMpsFIa`@","sGP@DiUj``mfHlZ","sFp@DiTvjhAlqEcP","Hid@@DjU^nBBH@FtaBXUMpq`XX@","sJP@DkVdAmQEb","qCp@AIZ`OH`","QMhDRVA@","qC`@ISJAlQE`","qCp@BOTAyhl","sJX@@eOU@ODB","sFp@DiTt@@AyaB","sGP@DkUj`FsDVMcMC@","Hid@@DjYUaBH`@FtaBXUMpqcHX@","qC`@Qz`OH@","HiD@@DjUVjj`AmHPfES\\LXZF@","sJP@H~j@[TQXqda`@","sJX@@eKU@OI@","sNp@Djejj@[\\QXu`","sJQ@@dsU@H","sJQ@@dkU@OI`","KAx@@YIMDVjh@vDHSBin@","Hid@@DjU^nBBD@FtaBXUMp`","sNp@DkgZj@[\\QXuqca`@","qC`@IRtDVqDVcEC@","Hed@@LdbRQeUUP@vTHSBinFP","sNp@DiUjj@P","qC`@IRtDT","sNp@DkYZj@[\\QXuqca`@","KAx@@IUkmUPAlHPfES]FDa`@","KAx@@IVjmUPAlHPfES]FNa`@","sOx@@drm\\@@@`","KAx@@ITjuUPAlHPfES]FBa`@","QMDARVAyH","sJP`@dfvhA@","HeT@@DjU_k``b`@[JDIaTwCLXfF@","KAx@@IToUUPAlHPfES]FJa`@","sGP@DkYj`FsDVMcEC@","qCb@AIZ`ODH","`I@`B","KAx@@IUzmUPAlHPfES]FFa`@","sNp@DkfZj@[\\QXu`","KAx@@ITnuUPAlHPfES]F@a`@","HiD@@LddURjjh@[RDIaTwB","sNp@Dj~Zj@[\\QXuqfa`@","Hed@@Dj{uZjj@FraBXUMpr","KAx@@ITsUUPAlHPfES\\","Hid@@LdbRQk``b@AmHPfES\\LXrF@","sOp@DjWkB@@GdH","sJQ@@dkU@OH@","Hid@@DjU^nBBH@FtaBXUMpqahX@","sGP@DiYj``mfHlZ","KAx@@IToUUPAlHPfES]FLa`@","qCp@AJZ`ODH","Hmt@@DjU]ZxHHj@AmhPfES\\Lj","sGP@DkUjPFsDVM@","qC`@IVtA@","Hed@@LdbJReUUP@vTHSBinFP","sNp@DjuZj@[\\QXuqea`@","KAx@@IUkmUPAlHPfES]FNa`@","HiD@@DkVUjj`AmHPfES\\H","Hed@@DkUeZjj@FraBXUMpr","sNp@DkVZj@[\\QXuqea`@","sJP@DiVhHKZbKFLLL@","HiD@@Djuyjj`AmHPfES\\H","sNp@DjUjj@[\\QXuq`a`@","HeT@@DjYUXPbH`@[JDIaTwCH","HiD@@DjwUjj`AmHPfES\\LXRF@","sNq@@djmUPB","KAx@@YIEEZjh@vDHSBincCPp@","sGP@Di^V`dmfHlZ","Hid@@DjYUaBHP@FtaBXUMp`","sNp@DjYjj@[\\QXuqba`@","sGP@Dkej`FsDVMcKC@","HeT@@DjU^k``b`@[JDIaTwCH","qC`@Qv`MbHmFBF@","sGQ@@djmT@`","qCr@XIKTAyH","qC`@IVtAlQEhpPp@","Hid@@LdbbQxXF@@AmHPfES\\LXjF@","sGP@DkYj`FsDVMcCC@","KAx@@IVsMUPAlHPfES\\","qCp@AIj`ODl","HiD@@DkeUjj`AmHPfES\\H","deT@@DjU[k``b`@vTHSBinFP","sJP@DkVdAmQEcFRF@","HiD@@LdbJTjjh@[RDIaTwCFDa`@","HiD@@DkYyjj`AmHPfES\\H","sJP@DjZhAyH","KAx@@IVkMUPAlHPfES]FDa`@","sJX@@dkU@OI@","Hed@@LdbRQUUUP@vTHSBinFXpLL@","Hed@@DjuUZjj@FraBXUMpr","sGP@Djfj`FsDVMcKC@","sNp@DkVZj@[\\QXuqba`@","sNp@DjyZj@[\\QXuqfa`@","qCb@AIj`OH@","sNp@DjUZj@[\\QXu`","KAx@@IWOMUPAlHPfES\\","Hid@@DjU^nBBH@D","Hed@@DjuvZjj@FraBXUMpr","sJP@DiVhHKZbKFLtL@","dmt@@DjU_ZxHHj@C[PaLJfxYVLKC@","sNp@DjuZj@[\\QXuqca`@","sJP@DkfhAmQEcFJF@","sNp@LdbJZj@[\\QXu`","HeT@@DjU_k``b`@[JDIaTwCLXFF@","KAx@@IVlmUPAlHPfES]FNa`@","HeT@@LdbbRKBDQD@CYPaLJfxYcEPp@","Hid@@DjUZnBBH@FtaBXUMpqcHX@","qCa@CIKTA@","HiD@@Dj~]jj`AmHPfES\\LXFF@","sKP@Di\\Zj@[TQX`","sGP@Djfj`FsDVMcEC@","HiD@@DkgYjj`AmHPfES\\H","sNp@DjuZj@[\\QXuqaa`@","KAx@@YIMDVjh@vDHSBincDPp@","sJP@DjVhHKZbKFLTL@","Hid@@LdbRQk``b@AmHPfES\\LXZF@","HiD@@Dj}Ujj`AmHPfES\\LXzF@","HeT@@DjU_k``bP@[JDIaTwCH","sNp@DkUZi@[\\QXu`","HiD@@DjYfjj`AmHPfES\\H","sGP@DjZj`FsDVMcAC@","Hmt@@DjU_jxHHj@AmhPfES\\Lj","Hid@@LdbRQk``R@AmHPfES\\H","KAx@@YIDUJjh@vDHSBincDPp@","qCr@XIKTAyD","sOq@@drm\\@@@|`@","Hed@@DjW^jjj@FraBXUMpsFBa`@","HeT@@DjY]zXFB@@[JDIaTwCH","Hkl@@DjU_Vk``bj`@[VDIaTwCJzX","Hid@@DjY}nBHH@FtaBXUMpqcHX@","sGX@@eKuT@|d@","sGP@Dj^Y`FsDVM@","HcL@@DjU_ZnBBJh@FqaBXUMprn`","sJP@DkVdAmQEcFJF@","sOq@@drm\\@@@|b@","sNp@DjyZj@[\\QXuqaa`@","HaD@@DjUZxHH@AyD@","qC`@Qv`H","dmt@@DjU_ZxHHj@C[PaLJfxYVLYC@","sGP@Dkej`FsDVMcMC@","Hed@@DjUUjjj@FraBXUMpsFHa`@","HeT@@LdbbRkBDQD@CYPaLJfxY@","KAx@@IU{MUPAlHPfES]FLa`@","RG@DTH","sJY@DDeVhA@","KAx@@YIDUJjh@vDHSBinc@Pp@","sJX@@dkU@OI`","sJQ@@dju@OI`","HeT@@LdbbRKBDQD@CYPaLJfxYcFPp@","sFp@DiTvjhAlqEcXpPp@","HaD@@DjUZxHH@AyG@","sNx@@eJ}UPB","sNp@LddUjj@[\\QXuqca`@","HaDH@@RVU[j@@@D","sNp@DkgZi@[\\QXu`","sGY@LDeVj`D","sNp@LdbJfZBZvxbqk@","sJP`@dfvhAyL","sGX@AddQjhAxe`","Hmt@@DjU_ZxHHj@AmhPfES\\LkFIa`@","qCh@CIKTA@","sNp@LdbLjj@[\\QXuq`a`@","sOq@@drm\\@@@|a@","KAx@@IUzmUPAlHPfES]FJa`@","sNx@AddQUUPB","sGP@Di]jP`mfHlZ","sJP`@TeZhA@","KAx@@IRjmUPHKXPaLJfx","HeT@@LdbRTM\\DDT@CYPaLJfxY@","HaF@@@Rfu[j@@@D","Hid@@DjYUaBH`@FtaBXUMpqchX@","KAx@@IUjmTpAlHPfES\\","Hid@@DjU^nBBD@FtaBXUMpqcHX@","sGP@DiUj``mfHl[FFF@","KAx@@IUvmUPAlHPfES]FLa`@","Hed@@LdbQTUUUP@vTHSBinFXqDL@","sJP@DkVhA@","sOx@@drm\\@@@|b@","KAx@@IUkMUPAlHPfES]FDa`@","HeT@@LdbRQU\\DDT@CYPaLJfxY@","HiD@@Dj}Yjj`AmHPfES\\LXrF@","HiD@@Dj{ujj`AmHPfES\\LXFF@","KAx@@IWNmUPAlHPfES]FFa`@","KAx@@IRkMUPHKXPaLJfx","sJP@DjYdAmQEcFZF@","sJY@LDeZhAyL","HaDH@@RVU[f@@@D","sJP`@deVhAyB","HaD@@DjWjZjj`AlHPfES\\","sGP@DkYj`FsDVMcMC@","sNp@DkgZj@[\\QXuqea`@","sJQ@@dlu@H","HeT@@DjU]k``b`@[JDIaTwCLXrF@","sJX@@dkU@OH`","RFDDQFCr`","sJP@DiYXIKZbKFLLL@","KAx@@YIHjjjh@vDHSBincGPp@","Hk\\@@DjU^ukmLHH@@@AmXPfES\\Lki`","sGQ@@djmT@|b@","Hid@@DjUfaBB`@FtaBXUMpqahX@","sNx@@eRmUPB","Hmt@@LdbRVak``ah@FvaBXUMprh","qCr@XIJtA@","KAx@@IWMmUPAlHPfES]FNa`@","HeT@@DjYYZPbJ@@[JDIaTwCH","sNp@DkfZj@[\\QXuqea`@","Hid@@DjU^nBAHAEVtaBXUMp`","dmt@@DjUgZDHJJ@C[PaLJfxYT","sGP@DkejPFsDVM@","sNx@@eJmUPB","qCb@AIf`H","HcL@@DjU_VnBBJh@FqaBXUMprnqcXX@","Hid@@DjUZnBBH@FtaBXUMpqahX@","sNp@LdbQZjBBvxbqkcGC@","sOx@@drm\\@@@|c@","sJP@H~j@^R@","KAx@@YIDcFjhDElHPfES\\","Hid@@DjUZnBAH@FtaBXUMp`","sNp@LddUji@[\\QXu`","sGP@DjfjPFsDVM@","HeT@@DjYUXPbD`@[JDIaTwCH","KAx@@IUoMUPAlHPfES]FDa`@","sFp@DiTt@@AyaD","Hed@@DjuuZjj@FraBXUMpsFIa`@","HeT@@DjUghP`h`@[JDIaTwCLXfF@","sOp@DjWkjj`FwDVM\\YhX@","sGP@Djfj`FsDVMcIC@","KAx@@IRkmUPHKXPaLJfzL]C@","sNx@@djmUPB","QM`AIdD","sOp@DjWkB@@Gbe@","sNp@DjyZj@[\\QXuqca`@","QM@HuAmd`","sNp@LddUjj@[\\QXuqea`@","daD@@DjUZxHD@CXPaLJfzLYC@","qCb@AIZPH","HiD@@LdbJTjjh@[RDIaTwCF@a`@","Hmt@@DjU_ZxHHi@AmhPfES\\Lj","HaDH@@RYWih@H@D","HiD@@LdbJTjjh@[RDIaTwCFHa`@","sGX@@djuT@|a@","sNp@DkfZj@[\\QXuqaa`@","Hid@@DjU^nBBH@GdL","KAx@@IVkMUPAlHPfES]FJa`@","qCr@XIKTAy@","HmT@@Dj{uVjjh@[ZDIaTwCJqaXX@","dmt@@DjYWVDHbJ@C[PaLJfxYT","Hif@@@RUe^Fh@@@P","HaDH@@Rfu[j@@@GdH","KAx@@IVsMUPAlHPfES]FDa`@","sKP@Di\\Zj@[TQXq`a`@","sJX@@eMU@OH@","HeT@@DjU^k``b`@[JDIaTwCLXFF@","Hmt@@LdbbRJXPbHh@FvaBXUMprh","sJP@DjvhAmQEcFBF@","dmt@@LdbbRNXPbHh@MmBDpj[aeXplL@","sJP`@dfvhAyD","sGP@Di^V`dmfHl[FVF@","KAx@@IVsmUPAlHPfES]FBa`@","sOq@@drm\\@@@|PP","sJY@BDeZhA@","HeT@@LdbRbmBDED@CYPaLJfxY@","Hed@@Djy[Zjj@FraBXUMpr","HeT@@DjU]k``b`@[JDIaTwCLXFF@","Hid@@DjUfaBB`@D","qCa@CIJtA@","QMPARVA@","Hid@@DjUfaBB`@FtaBXUMpqcHX@","sJY@BDfZhA@","HeT@@DjUghP`hP@[JDIaTwCH","Hed@@Dj{uZjj@FraBXUMpsFIa`@","dmt@@DjYUvDHbJ@C[PaLJfxYVLUC@","sNp`@dfuZj@P","sJQ@@dmU@OH@","sJX@@dmU@H","HeT@@DjU]k``b`@[JDIaTwCLXZF@","HiD@@LdfbJZjh@[RDIaTwCFAa`@","sOx@@drm\\@@@|a@","deT@@DjUfhP`h`@vTHSBinFP","dmt@@DjU]ZxHHj@C[PaLJfxYVLKC@","sOp@DjWkjj`FwDVM\\XHX@","HcL@@LdbbRNSBDQEP@McBDpj[ae]cFpp@","HiD@@Dj}Yji`AmHPfES\\H","HaDH@@RYe[hB@@D","Hid@@DjU^njjj@FtaBXUMpq`XX@","deT@@LdbbRKBDQB@FraBXUMpr","QMPARZA@","sOq@@drm\\@@@|QX","HaD@@DjYvxH`@A@","HcL@@LdbbRNcBDQEP@McBDpj[ae]@","QMhDRZA@","RG@DXLHmP","QM`BN`XQYd","RG@DTLHmP","QMHAIXFEVd","QMDARVAaH","RFPDXLHmP","RF@Q`vRbdLEC@","RF@QpvR@","QO@HyjAmd`","`II@B","`II@CFspqJp","`II@CF[@hM@prB`","`H@[T[|B`XN@PdM@p|@bHrBcDk@","RG@DXMj}F@","QM`BN`[L~b@","RG@DTMj}D@","QMHAIXFt~j@","QMDARVA}L@","RFPDXMj}D@","sKP@Di\\YZ@[TQXqaa`@","eF`BHD"]);},n.qu=function(t,A){var e,r;if(!A)return null;for(r=B(Q.S7,H,6,(n.nu.length+31)/32|0,15,1),A=n.vu(A),n.cu(t.g,A),e=0;e<n.nu.length;e++)n._t(t.g,n.ou[e]),n.Pt(t.g,1,4)>0&&(r[e/32|0]|=1<<31-e%32);return r;},n.ru=function(t,A){var e,r;if(!A)return null;for(r=B(Q.T7,e0,6,(n.nu.length+63)/64|0,14,1),A=n.vu(A),n.cu(t.g,A),e=0;e<n.nu.length;e++)n._t(t.g,n.ou[e]),n.Pt(t.g,1,4)>0&&(r[e/64|0]=Pt(r[e/64|0],c1(1,63-e%64)));return r;},n.su=function(){var t,A;if(n.ou==null)for(A=new n._q(false),n.ou=B(Q.S8,GA,26,n.nu.length,0,1),t=0;t<n.nu.length;t++)n.ou[t]=n.Pq(A,n.nu[t]),n.bw(n.ou[t],1);},n.tu=function(t){var A;if(t.f!=null){for(A=0;A<t.f.length;A++)if((t.c[A]&~t.f[A])!=0)return  false;}else return  false;return n.uu(t);},n.uu=function(t){return !t.d&&(t.d=n.Rq(new n._q(false),t.e)),!t.a&&(t.a=n.Rq(new n._q(false),t.b)),n.cu(t.g,t.d),n._t(t.g,t.a),n.Wt(t.g);},n.vu=function(t){var A,e;if(t.K){for(A=0;A<t.q;A++)if(Rt(z(t.B[A],i0),0)){for(t=new n.vw(t),e=A;e<t.q;e++)Rt(z(t.B[e],i0),0)&&(t.C[e]=-1);n.vl(t);}}return t;},n.wu=function(t,A,e){t.b=null,t.a=A,e==null?t.c=n.qu(t,A):t.c=e;},n.xu=function(t,A,e){t.e=null,t.d=A,e==null?t.f=n.qu(t,A):t.f=e;},n.yu=function(){n.pu(),this.g=new n.iu(),n.su();},n.zu=function(t){n.pu();var A,e,r,P,i,D;if(t==null)return null;for(A=B(Q.O7,Y,6,t.length*8,15,1),r=0;r<t.length;r++)for(D=t[r],P=7;P>=0;P--)e=D&15,e>9&&(e+=7),A[r*8+P]=48+e<<24>>24,D>>=4;return h.akb((i=A,O1.Enb(),i));},n.Au=function(t){n.pu();var A,e,r,P;if(w.UAb(t).length==0||(w.UAb(t).length&7)!=0)return null;for(r=B(Q.S7,H,6,w.UAb(t).length/8|0,15,1),e=0;e<w.UAb(t).length;e++)P=e/8|0,A=(w.PAb(e,w.UAb(t).length),w.UAb(t).charCodeAt(e)-48),A>16&&(A-=7),r[P]<<=4,r[P]+=A;return r;},n.Bu=function(t,A){n.pu();var e,r,P,i;for(i=0,r=0,P=0,e=0;e<t.length;e++)i+=rn(t[e]&A[e]),r+=rn(t[e]),P+=rn(A[e]);return i/d.Math.sqrt(r*P);},n.Cu=function(t,A){n.pu();var e,r,P;for(P=0,e=0,r=0;r<t.length;r++)P+=rn(t[r]&A[r]),e+=rn(t[r]|A[r]);return P/e;},I(180,1,{},n.yu),Q.G8=N(180),n.Eu=function(t){t.j=1,t.k=10,t.n=400,t.f=400,t.i=true,t.d="black",t.b=new s.Mj(),t.a=new s.Mj(),t.c=new h.Mkb(),t.e=new R1.Lgb(12);},n.Fu=function(t,A){var e,r;return e=(r=R1.Jgb(t.e,A),new B0.bhb(r)).b,e;},n.Gu=function(t){return t.k;},n.Hu=function(t){var A,e,r,P,i,D;if(i='<svg id="'+n.Ou(t.g!=null?t.g:"mol"+n.Du)+Y1+'xmlns="http://www.w3.org/2000/svg" version="1.1" width="'+t.n+'px" height="'+t.f+'px" viewBox="0 0 '+t.n+" "+t.f+`">
`,D=t.i?"<style> #"+n.Ou(t.g!=null?t.g:"mol"+n.Du)+" { pointer-events:none; } #"+n.Ou(t.g!=null?t.g:"mol"+n.Du)+` .event  { pointer-events:all; } line { stroke-linecap:round; } polygon { stroke-linejoin:round; } </style>
`:'<g style="font-size:'+t.k+'px; fill-opacity:1; stroke-opacity:1; fill:black; stroke:black; font-weight:normal; text-rendering:optimizeLegibility; font-family:sans-serif; stroke-linejoin:round; stroke-linecap:round; stroke-dashoffset:0;">',i+="  ",i+=D,t.i){for(P=new s.trb(t.b);P.a<P.c.a.length;)r=s.srb(P),n.Iu(t,r);for(e=new s.trb(t.a);e.a<e.c.a.length;)A=s.srb(e),n.Iu(t,A);}return t.i||n.Iu(t,"</g>"),i+t.c.a+"</svg>";},n.Iu=function(t,A){h.Kkb(t.c,"  "),h.Kkb(t.c,A),h.Kkb(t.c,m1);},n.Ju=function(t,A,e){n._c(),n.de.call(this,t,A),n.Eu(this),this.g=e,++n.Du;},n.Ou=function(t){var A,e,r,P,i;for(i=new h.Mkb(),P=w.UAb(t).length,r=0;r<P;r++)switch(A=aD(t,r,w.UAb(t).length),e=A&n1,A){case 38:i.a+="&amp;";break;case 60:i.a+="&lt;";break;case 62:i.a+="&gt;";break;case 34:i.a+="&quot;";break;case 39:i.a+="&apos;";break;default:i.a+=String.fromCharCode(e);}return i.a;},I(259,121,{121:1},n.Ju),L.qb=function(t){var A;A='<line x1="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.a),(p.vnb(),p.rnb))))+Y1+'y1="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.c),p.rnb)))+Y1+'x2="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.b),p.rnb)))+Y1+'y2="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.d),p.rnb)))+Y1+yP+this.d+Y1+fr+this.j+Dr,n.Iu(this,A);},L.rb=function(t){var A,e;A=h.Nib(p.mlb(p.ilb(new p.nlb(3*this.j),(p.vnb(),p.rnb)))),e='<line stroke-dasharray="'+A+","+A+Y1+'x1="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.a),p.rnb)))+Y1+'y1="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.c),p.rnb)))+Y1+'x2="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.b),p.rnb)))+Y1+'y2="'+h.Nib(p.mlb(p.ilb(new p.nlb(t.d),p.rnb)))+Y1+yP+this.d+Y1+"stroke-width:"+this.j+Dr,n.Iu(this,e);},L.sb=function(t){var A,e;for(e=new h.Okb('<polygon points="'),A=0;A<t.a;A++)h.Ikb(e,w1(d.Math.round(t.b[A]))),e.a+=",",h.Ikb(e,w1(d.Math.round(t.c[A]))),e.a+=" ";h.Kkb(e,'" fill="'+this.d+Y1+yP+this.d+Y1+fr+this.j+Dr),n.Iu(this,e.a);},L.tb=function(t,A,e){var r,P,i,D;i=(r=(D=R1.Jgb(this.e,t),new B0.bhb(D)).b,r),P='<text x="'+h.Nib(p.mlb(p.ilb(new p.nlb(A-i/2),(p.vnb(),p.rnb))))+Y1+'y="'+h.Nib(p.mlb(p.ilb(new p.nlb(e+this.k/3),p.rnb)))+Y1+'stroke="none" font-size="'+this.e.b+Y1+'fill="'+n.Ou(this.d)+'">'+n.Ou(t)+"</text>",n.Iu(this,P);},L.ub=function(t,A,e){var r;r='<circle cx="'+h.Nib(p.mlb(p.ilb(new p.nlb(t+e/2),(p.vnb(),p.rnb))))+Y1+'cy="'+h.Nib(p.mlb(p.ilb(new p.nlb(A+e/2),p.rnb)))+Y1+'r="'+h.Nib(p.mlb(p.ilb(new p.nlb(e/2),p.rnb)))+Y1+'fill="'+this.d+Dr,n.Iu(this,r);},L.vb=function(t){return n.Fu(this,t);},L.wb=function(){return n.Gu(this);},L.yb=function(t,A,e,r){var P,i;P=n.Ll(this.L,t),i='<circle id="'+n.Ou(this.g!=null?this.g:"mol"+n.Du)+":Atom:"+t+Y1+$8+(P==0?"":'data-atom-map-no="'+P+Y1)+'cx="'+h.Nib(p.mlb(p.ilb(new p.nlb(e),(p.vnb(),p.rnb))))+Y1+'cy="'+h.Nib(p.mlb(p.ilb(new p.nlb(r),p.rnb)))+Y1+'r="8'+Y1+a8,s.sj(this.a,i);},L.zb=function(t,A,e,r,P){var i;i='<line id="'+n.Ou(this.g!=null?this.g:"mol"+n.Du)+":Bond:"+t+Y1+$8+'x1="'+h.Nib(p.mlb(p.ilb(new p.nlb(A),(p.vnb(),p.rnb))))+Y1+'y1="'+h.Nib(p.mlb(p.ilb(new p.nlb(e),p.rnb)))+Y1+'x2="'+h.Nib(p.mlb(p.ilb(new p.nlb(r),p.rnb)))+Y1+'y2="'+h.Nib(p.mlb(p.ilb(new p.nlb(P),p.rnb)))+Y1+fr+8+Y1+a8,s.sj(this.b,i);},L.Ab=function(t){this.j=nn(w1(d.Math.round(100*d.Math.max(t,1))))/100;},L.Bb=function(t){this.d="rgb("+((t&W0)>>16)+","+((t&Y0)>>8)+","+(t&255)+")";},L.Cb=function(t){this.k!=t&&(this.k=t,this.e=new R1.Lgb(t));},L.Db=function(t,A){return this.n=zt(w1(d.Math.round(t.b))),this.f=zt(w1(d.Math.round(t.a))),n.$d(this,t,A);},L.pb=function(){return n.Hu(this);},L.f=0,L.i=false,L.j=0,L.k=0,L.n=0,n.Du=0,Q.I8=N(259),n.Zu=function(t,A){var e;if(t.c!=null){for(e=0;e<t.c.length;e++)if(A==t.c[e]&&t.d[e])return  true;}return  false;},n.$u=function(t,A,e,r,P){this.a=t,this.b=A,this.g=e,this.f=r,this.e=P;},I(184,1,{184:1},n.$u),L.a=0,L.b=0,L.e=false,L.f=false,L.g=0,Q.M8=N(184),n._u=function(t,A){!t.b&&(t.b=new Ut.S4()),Ut.M4(t.b,h.sjb(A));},n.av=function(t,A,e,r){var P,i,D,u,o,l;if(P=n.Uk(A,t.d),n.jn(A,P,t.e),n.rn(A,P,t.o,false),n.gn(A,P,t.a),Rt(t.c,0)&&(Rt(z(t.c,2),0)?(t.c=z(t.c,-3),A.u[P]|=r0):A.u[P]&=ae,n.vn(A,P,t.c,true)),t.b){for(l=B(Q.S7,H,6,t.b.d.a.length,15,1),o=0;o<t.b.d.a.length;o++)l[o]=Ut.O4(t.b,o).a;n.pn(A,P,l),w.mAb(t.b.d.a,0);}else if(String.fromCharCode(e).toLowerCase().charCodeAt(0)==e&&h.cib(e)){if(t.d!=5&&t.d!=6&&t.d!=7&&t.d!=8&&t.d!=15&&t.d!=16&&t.d!=33&&t.d!=34)throw Lt(new h.cC("SmilesParser: atomicNo "+t.d+" must not be aromatic. Position:"+(r-1)));A.u[P]|=r0;}else A.u[P]&=ae;if(t.f)for(u=new s.trb(t.f);u.a<u.c.a.length;)D=s.srb(u),t.r=true,n.Cn(D,0,0),n.On(A,true),n.dl(A,D,P);return t.g!=-1&&t.d!=1&&(i=B(Q.O7,Y,6,1,15,1),i[0]=t.g<<24>>24,n.nn(A,P,i)),P;},n.bv=function(t,A){var e;for(e=0;A<t.length&&(t[A]!=93||e!=0);)t[A]==91?++e:t[A]==93&&--e,++A;if(A==t.length)throw Lt(new h.cC("SmilesParser: No matching closing bracket found."));return A+1;},n.cv=function(t,A,e){var r,P;if(A<3||(r=A-1,(t[r]==43||t[r]==45)&&--r,!AA(t[r]&n1))||(--r,t[r]!=68&&t[r]!=82&&t[r]!=88&&t[r]!=122))return  false;for(e[0]=0;r>0&&tu(t[r-1]);)--r,++e[0];for(P=A+1;tu(t[r]);){if(t.length<=P||t[r]!==t[P])return  false;++r,++P;}return  true;},n.dv=function(t,A,e,r,P,i){var D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O;if(A[e-1]==36){if(t.r=true,t.q=new s.Mj(),e+=n.hv(t,A,e-1,t.q)-1,A[e++]!=93){if(!i)throw Lt(new h.cC("SmilesParser: A positive recursive SMARTS followed by another one or by atom query features is not supported. Position:"+(e-1)));(t.k&128)==0&&(e=n.bv(A,e));}return e;}if(A[e-1]==42)t.d=6,t.c=Pt(t.c,1);else if(A[e-1]==63)t.d=0;else if(l=A[e-1]==33,l&&(t.r=true,t.c=Pt(t.c,1),++e),A[e-1]==82&&P&&(AA(A[e]&n1)||t.j&&A[e]==123))t.d=6,t.c=Pt(t.c,1),--e,l&&--e;else {if(U=new n.jv(),!n.ev(t,A,e-1,r,U))throw Lt(new h.cC("SmilesParser: Unexpected character in atom definition:'"+String.fromCharCode(A[e-1]&n1)+ti+(e-1)));if(t.d=U.a,e+=U.b-1,(t.k&3)!=2&&(t.g=9),P&&(A[e]==44||l)){for(c=U.d,T=U.c,R=e-U.b;R<r;){if(!n.ev(t,A,R,r,U)){if(!l)throw Lt(new h.cC("SmilesParser: Unexpected character in atom list:'"+String.fromCharCode(A[R]&n1)+Ai+R));break;}if(U.a==1){if(!l)throw Lt(new h.cC("SmilesParser: Hydrogen is not supported in positive atom lists:'"+h._jb((s.yrb(A,R,r),s.Drb(A,B(Q.O7,Y,6,r-R,15,1),R,r)))+Ai+R));}else n._u(t,U.a),c=c|U.d,T=T|U.c;if(R+=U.b,l&&A[R]!=59&&A[R]!=38||!l&&A[R]!=44||l&&A[R+1]!=33)break;++R,A[R]==33&&++R;}t.b&&t.b.d.a.length>1&&(t.g=-1,T?c||(t.c=Pt(t.c,4)):t.c=Pt(t.c,2)),e=R;}}for(g=new n.nv(A),F=B(Q.S7,H,6,1,15,1),S=true;S;){if(A[e]==64){++e,A[e]==64&&(t.i=true,++e),t.p=true;continue;}if(A[e]==58){for(++e;AA(A[e]&n1);)t.o=10*t.o+A[e]-48,++e;continue;}if(A[e]==91)throw Lt(new h.cC("SmilesParser: nested square brackets found. Position:"+e));if(A[e]==93){++e,S=false;continue;}if(t.e=n.gv(A,e,F),F[0]!=0){e+=F[0],t.e==0&&(t.c=Pt(t.c,re));continue;}if(o=A[e]==33,o&&++e,A[e]==72){++e,e+=n.kv(g,e,1),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,128)),g.c<=1&&g.b>=1&&(D=Pt(D,256)),g.c<=2&&g.b>=2&&(D=Pt(D,512)),g.c<=3&&g.b>=3&&(D=Pt(D,f0)),o?(t.c=Pt(t.c,D),t.g=-1):g.b==g.c?t.g=g.c:(t.c=Pt(t.c,z(X0,tA(D))),t.g=-1);continue;}if(A[e]==68||A[e]==100){++e,e+=n.kv(g,e,1),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,r0)),g.c<=1&&g.b>=1&&(D=Pt(D,yA)),g.c<=2&&g.b>=2&&(D=Pt(D,sA)),g.c<=3&&g.b>=3&&(D=Pt(D,TA)),g.c<=4&&g.b>=4&&(D=Pt(D,j0)),nt(D,0)!=0&&(o?t.c=Pt(t.c,D):Rt(z(t.c,aA),0)?t.c=z(t.c,tA(D)):(D=en(D,aA),t.c=Pt(t.c,D)));continue;}if(A[e]==122&&t.j){++e,e+=n.kv(g,e,4),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,Mn)),g.c<=1&&g.b>=1&&(D=Pt(D,ni)),g.c<=2&&g.b>=2&&(D=Pt(D,ei)),g.c<=3&&g.b>=3&&(D=Pt(D,ri)),g.c<=4&&g.b>=4&&(D=Pt(D,fe)),nt(D,0)!=0&&(o?t.c=Pt(t.c,D):Rt(z(t.c,Un),0)?t.c=z(t.c,tA(D)):(D=en(D,Un),t.c=Pt(t.c,D)));continue;}if(A[e]==88){if(++e,e+=n.kv(g,e,1),O=(n.Rk(),n.Mk)[t.d],O==null)continue;E=O[0],y=n.gv(A,e,F),F[0]!=0&&(n.qo(t.d)?E+=y:t.d==6?E-=d.Math.abs(y):E-=y),D=0,E-g.c<=0&&E-g.b>=0&&(D=Pt(D,N0)),E-g.c<=1&&E-g.b>=1&&(D=Pt(D,UA)),E-g.c<=2&&E-g.b>=2&&(D=Pt(D,H1)),nt(D,0)!=0&&(o?t.c=Pt(t.c,D):Rt(z(t.c,$A),0)?t.c=z(t.c,tA(D)):(D=en(D,$A),t.c=Pt(t.c,D)));continue;}if(A[e]==65||A[e]==97){t.c=Pt(t.c,o^A[e]==65?4:2),++e;continue;}if(A[e]==82){++e,e+=n.kv(g,e,3),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,8)),g.c<=1&&g.b>=1&&(D=Pt(D,16)),g.c<=2&&g.b>=2&&(D=Pt(D,32)),g.c<=3&&g.b>=3&&(D=Pt(D,64)),g.b>3&&n.Hv(t.n,(o?"!R":"R")+g.b),nt(D,0)!=0&&(o?t.c=Pt(t.c,D):Rt(z(t.c,120),0)?t.c=z(t.c,tA(D)):(D=en(D,120),t.c=Pt(t.c,D)));continue;}if(A[e]==114){if(++e,e+=n.kv(g,e,1),g.a){o?t.c=Pt(t.c,384):t.c=Pt(t.c,8);continue;}k=g.c,g.b>g.c&&n.Hv(t.n,(o?"!r":"r")+("{"+g.c+"-"+g.b+"}")),!o&&k>=3&&k<=7?t.c=Pt(t.c,Q.Mfb(k<<22)):g.b>g.c||n.Hv(t.n,(o?"!r":"r")+k);continue;}if(A[e]==118){++e,e+=n.kv(g,e,1),E=g.c,g.b>g.c&&n.Hv(t.n,(o?"!v":"v")+("{"+g.c+"-"+g.b+"}")),!o&&E<=14?t.a=E:g.b>g.c||n.Hv(t.n,(o?"!v":"v")+E);continue;}if(A[e]==94){if(++e,u=A[e++]-48,u<1||u>3)throw Lt(new h.cC("SmilesParser: Unsupported hybridization. Position:"+e));b=u==1?H1:u==2?UA:N0,o||(b=z($A,tA(b))),t.c=Pt(t.c,b);continue;}if(A[e]==36){if(!o)throw Lt(new h.cC("SmilesParser: non-negated recursive SMARTS relating to preceding atom are not supported yet. Position:"+e));!t.f&&(t.f=new s.Mj()),t.r=true,e+=n.hv(t,A,e,t.f);continue;}if(P&&(A[e]==59||A[e]==38)){t.r=true,++e;continue;}if(P&&A[e]==44&&n.cv(A,e,F)){t.r=true,e+=F[0]+1;continue;}if(P&&A[e]==44&&(t.k&128)!=0){t.r=true,e+=1;break;}throw A[e]==44?Lt(new h.cC("SmilesParser: alternative atom definitions not supported. (Tip: enumerate SMARTS): '"+String.fromCharCode(A[e]&n1)+TP+e)):Lt(new h.cC("SmilesParser: unexpected character inside brackets: '"+String.fromCharCode(A[e]&n1)+TP+e));}return e;},n.ev=function(t,A,e,r,P){var i,D;if(P.d=true,P.c=true,A[e]==35){for(++e,t.r=true,P.a=0,P.b=1;e<r&&AA(A[e]&n1);)P.a=10*P.a+A[e]-48,++P.b,++e;if(P.a==0||P.a>=(n.Rk(),n.Lk).length)throw Lt(new h.cC("SmilesParser: Atomic number out of range. position:"+(e-1)));return  true;}if(A[e]>=65&&A[e]<=90){if(P.b=A[e+1]>=97&&A[e+1]<=122?2:1,P.a=(n.Rk(),n.po(h.zkb(w.sAb(A,e,(D=P.b,O1.Enb(),D))),321)),P.b==2&&P.a==0&&(P.b=1,P.a=n.po(h.zkb(w.sAb(A,e,(i=P.b,i))),321)),P.d=false,P.a==0)throw Lt(new h.cC("SmilesParser: Unknown atom label. position:"+(e-1)));return  true;}return A[e]==65&&A[e+1]==115||A[e]==83&&A[e+1]==101?(P.b=2,P.a=(n.Rk(),n.po(h.zkb(w.sAb(A,e,(i=P.b,O1.Enb(),i))),321)),P.c=false,true):A[e]==99||A[e]==110||A[e]==111||A[e]==112||A[e]==115?(P.b=1,P.a=(n.Rk(),n.po(h.zkb(w.sAb(A,e,(i=P.b,O1.Enb(),i))),321)),P.c=false,true):false;},n.fv=function(t,A,e,r,P){if(A[e-1]==42)t.d=6,t.c=Pt(t.c,1);else if(A[e-1]==63)t.d=0;else if((A[e-1]==65||A[e-1]==97)&&P)t.d=6,t.c=Pt(t.c,1),t.c=Pt(t.c,A[e-1]==65?4:2),t.r=true;else switch(String.fromCharCode(A[e-1]).toUpperCase().charCodeAt(0)){case 66:e<r&&A[e]==114?(t.d=35,++e):t.d=5;break;case 67:e<r&&A[e]==108?(t.d=17,++e):t.d=6;break;case 70:t.d=9;break;case 73:t.d=53;break;case 78:t.d=7;break;case 79:t.d=8;break;case 80:t.d=15;break;case 83:t.d=16;}return e;},n.gv=function(t,A,e){var r,P;if(e[0]=0,t[A]==43||t[A]==45){for(P=t[A],r=1,++e[0];t[A+e[0]]==P;)++r,++e[0];return r==1&&AA(t[A+1]&n1)&&(r=t[A+1]-48,++e[0]),P==43?r:-r;}return 0;},n.hv=function(t,A,e,r){var P,i,D,u,o;if(A[e+1]!=40)throw Lt(new h.cC("SmilesParser: '$' for recursive SMARTS must be followed by '('. position:"+e));for(u=1,i=e+2;i<A.length&&u>0;)A[i]==40?++u:A[i]==41&&--u,++i;if(u>0)throw Lt(new h.cC("SmilesParser: Missing closing ')' for recursive SMARTS. '('-position:"+(e+1)));if(D=new n.uw(16,16),o=new n.Jv(t.k),n.Fv(o,t.n.f),n.xv(o,D,A,e+2,i-1,true,true),n.On(D,true),w.kAb(r.a,D),A[e-1]==33)for(P=0;P<D.q;P++)n.vn(D,P,i0,true);return i-e;},n.iv=function(t,A){this.n=t,this.k=A,this.j=(A&16)==0,this.d=-1,this.e=0,this.o=0,this.a=-1,this.g=-1,this.c=0;},I(283,1,{},n.iv),L.a=0,L.c=0,L.d=0,L.e=0,L.g=0,L.i=false,L.j=false,L.k=0,L.o=0,L.p=false,L.r=false,Q.L8=N(283),n.jv=function(){this.a=-1;},I(370,1,{},n.jv),L.a=0,L.b=0,L.c=false,L.d=false,Q.J8=N(370),n.kv=function(t,A,e){var r,P,i,D,u;if(t.a=false,t.d=A,AA(t.e[A]&n1)){for(u=n.lv(t),t.c=t.b=u,r=A-1;r>1&&n2(t.e[r-1]);)--r;for(;t.e[t.d]==44;){for(D=true,i=A-r,P=0;P<i;P++)if(t.e[r+P]!==t.e[t.d+1+P]){D=false;break;}if(!D)break;t.d+=1+i,u=n.lv(t),t.c>u?t.c=u:t.b<u&&(t.b=u);}return t.d-A;}return t.e[A]==123&&AA(t.e[A+1]&n1)?(++t.d,t.c=n.lv(t),t.e[t.d++]!=45||!AA(t.e[t.d]&n1)||(t.b=n.lv(t),t.e[t.d++]!=125)?0:t.d-A):(t.c=1,t.b=e,t.a=true,0);},n.lv=function(t){var A;return A=t.e[t.d++]-48,AA(t.e[t.d]&n1)&&(A=10*A+(t.e[t.d++]-48)),A;},n.mv=function(t){return "{"+t.c+"-"+t.b+"}";},n.nv=function(t){this.e=t;},I(371,1,{},n.nv),L.pb=function(){return n.mv(this);},L.a=false,L.b=0,L.c=0,L.d=0,Q.K8=N(371),n.pv=function(t,A,e,r,P,i){var D;return D=n.Wk(t.k,A,e,r),s.tob(t.s,A+"_"+e,j(v(Q.S7,1),H,6,15,[P,i])),s.tob(t.s,e+"_"+A,j(v(Q.S7,1),H,6,15,[P,i])),s.ssb(t.a,A),s.ssb(t.a,e),D;},n.qv=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b;for(l=B(Q.S7,H,6,t.k.f,15,1),u=B(Q.S7,H,6,t.k.f,15,1),o=B(Q.S7,H,6,t.k.f,15,1),U=B(Q.S7,H,6,t.k.f,15,1),e=n.Zl(t.k,0,A),r=n.Zl(t.k,1,A),u[0]=e,u[1]=r,o[0]=-1,o[1]=A,l[e]=1,l[r]=2,U[e]=-1,U[r]=e,D=1,y=1;D<=y&&l[u[D]]<15;){for(b=u[D],T=0;T<n.$o(t.k,b);T++)if(P=n.Zo(t.k,b,T),P!=U[b]){if(i=n._o(t.k,b,T),P==e){for(o[0]=i,c=0;c<=y;c++)t.g[o[T]]||(t.g[o[T]]=true,++t.c);return;}n.Km(t.k,P)&&l[P]==0&&(++y,u[y]=P,o[y]=i,l[P]=l[b]+1,U[P]=b);}++D;}},n.rv=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c;for(n.bw(t.k,7),y=false,T=B(Q.S7,H,6,2,15,1),c=B(Q.S7,H,6,2,15,1),U=B(Q.S7,H,6,2,15,1),r=0;r<t.k.g;r++)if(!n.jq(t.k,r)&&n.im(t.k,r)==2){for(D=0;D<2;D++){for(T[D]=-1,U[D]=-1,A=n.Zl(t.k,D,r),l=0;l<n.$o(t.k,A);l++)P=n._o(t.k,A,l),P!=r&&(T[D]==-1&&(n.im(t.k,P)==257||n.im(t.k,P)==129)?(T[D]=n.Zo(t.k,A,l),c[D]=P):U[D]=n.Zo(t.k,A,l));if(T[D]==-1)break;}if(T[0]!=-1&&T[1]!=-1){for(o=n.im(t.k,c[0])==n.im(t.k,c[1]),u=0;u<2;u++)T[u]==n.Zl(t.k,0,c[u])&&(o=!o);for(i=0;i<2;i++)U[i]!=-1&&U[i]<T[i]&&(o=!o);n.Jn(t.k,r,o?2:1,false),y=true;}}for(e=0;e<t.k.g;e++)(n.im(t.k,e)==257||n.im(t.k,e)==129)&&n.Mn(t.k,e,1);return y;},n.sv=function(t){return t==61?2:t==35?4:t==36?32:t==58?8:t==62?16:t==126?127:1;},n.tv=function(t,A){var e,r,P,i,D,u,o,l,U,y,T;for(e=0;e<t.k.f;e++)if(n.Km(t.k,e))for(o=B(Q.S7,H,6,t.k.f,15,1),u=B(Q.S7,H,6,t.k.f,15,1),l=B(Q.S7,H,6,t.k.f,15,1),u[0]=e,o[e]=1,l[e]=-1,D=0,U=0;D<=U;){for(P=(o[u[D]]&1)==1?1:2,y=0;y<n.$o(t.k,u[D]);y++)if(r=n._o(t.k,u[D],y),n.fm(t.k,r)==P&&A[r]&&(i=n.Zo(t.k,u[D],y),o[i]==0)){if(P==1&&n.Km(t.k,i)){for(T=u[D];T!=-1;)n.Mn(t.k,n.Xo(t.k,i,T),P==1?2:1),P=3-P,i=T,T=l[T];return n.sn(t.k,e,false),n.sn(t.k,i,false),t.b-=2,true;}u[++U]=i,l[i]=u[D],o[i]=o[u[D]]+1;}++D;}return  false;},n.uv=function(t){var A,e,r,P;for(A=0;A<t.k.f;A++)if(n.Tl(t.k,A)==7&&n.Bl(t.k,A)==0&&n.vp(t.k,A)>3&&n.Qo(t.k,A)>0){for(P=0;P<n.$o(t.k,A);P++)if(e=n.Zo(t.k,A,P),r=n._o(t.k,A,P),n.fm(t.k,r)>1&&n.Hm(t.k,e)){n.im(t.k,r)==4?n.Mn(t.k,r,2):n.Mn(t.k,r,1),n.jn(t.k,A,n.Bl(t.k,A)+1),n.jn(t.k,e,n.Bl(t.k,e)-1),n.gn(t.k,A,-1);break;}}},n.vv=function(t){return t.r?"Unresolved SMARTS features:"+t.r:"";},n.wv=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt;for(n.bw(t.k,1),t.g=B(Q.xfb,Tt,6,t.k.g,16,1),t.c=0,y=0;y<t.k.g;y++)n.im(t.k,y)==8&&(n.Mn(t.k,y,1),t.g[y]=true,++t.c);for(R=B(Q.xfb,Tt,6,t.k.f,16,1),Nt=new n.Dt(t.k,3),S=B(Q.xfb,Tt,6,Nt.j.a.length,16,1),Vt=0;Vt<Nt.j.a.length;Vt++){for(Ht=s.xj(Nt.j,Vt),S[Vt]=true,G=Ht,At=0,it=G.length;At<it;++At)if(O=G[At],!n.Km(t.k,O)){S[Vt]=false;break;}if(S[Vt]){for(J=Ht,ot=0,lt=J.length;ot<lt;++ot)O=J[ot],R[O]=true;for(Jt=s.xj(Nt.k,Vt),x=Jt,V=0,K=x.length;V<K;++V)O=x[V],t.g[O]||(t.g[O]=true,++t.c);}}for(T=0;T<t.k.g;T++)!t.g[T]&&(Nt.c[T]&n1)!=0&&n.Km(t.k,n.Zl(t.k,0,T))&&n.Km(t.k,n.Zl(t.k,1,T))&&n.qv(t,T);for(c=0;c<t.k.g;c++)t.g[c]||(i=n.Zl(t.k,0,c),D=n.Zl(t.k,1,c),!R[i]&&!R[D]&&n.Km(t.k,i)&&n.Km(t.k,D)&&(t.g[c]=true,++t.c));for(n.bw(t.k,7),(t.q==2||t.q==1&&t.p)&&n.Dv(t),F=B(Q.xfb,Tt,6,t.k.g,16,1),t.k.g>=0&&h.Skb(t.g,0,F,0,t.k.g),Et=0;Et<Nt.j.a.length;Et++)if(S[Et]){for(Ht=s.xj(Nt.j,Et),Dt=Ht,Qt=0,bt=Dt.length;Qt<bt;++Qt)if(ut=Dt[Qt],!n.Ev(t,ut))for(n.Km(t.k,ut)&&(n.sn(t.k,ut,false),--t.b),O=0;O<n.$o(t.k,ut);O++)g=n._o(t.k,ut,O),t.g[g]&&(t.g[g]=false,--t.c);}for(n.Cv(t),ct=0;ct<Nt.j.a.length;ct++)if(S[ct]&&s.xj(Nt.k,ct).length==6){for(Jt=s.xj(Nt.k,ct),E=true,o=Jt,l=0,U=o.length;l<U;++l)if(u=o[l],!t.g[u]){E=false;break;}E&&(n.Bv(t,Jt[0]),n.Bv(t,Jt[2]),n.Bv(t,Jt[4]),n.Cv(t));}for(wt=5;wt>=4;wt--)do for(st=false,u=0;u<t.k.g;u++)if(t.g[u]){for(r=0,k=0;k<2;k++)for(b=n.Zl(t.k,k,u),O=0;O<n.$o(t.k,b);O++)t.g[n._o(t.k,b,O)]&&++r;if(r==wt){n.Bv(t,u),n.Cv(t),st=true;break;}}while(st);for(;t.b>=2&&n.tv(t,F););if(A){if(t.b!=0)for(P=0;P<t.k.f;P++)n.Km(t.k,P)&&(n.sn(t.k,P,false),n.vn(t.k,P,2,true),--t.b,e=true);if(t.c!=0)for(u=0;u<t.k.g;u++)t.g[u]&&(t.g[u]=false,n.Mn(t.k,u,8),--t.c,e=true);}else for(P=0;P<t.k.f;P++)n.Km(t.k,P)&&n.op(t.k,P)!=0&&(n.sn(t.k,P,false),n.wn(t.k,P,32),--t.b);if(t.q==0||t.q==1&&!e){if(t.b!=0)throw Lt(new h.cC(Pi));if(t.c!=0)throw Lt(new h.cC(Pi));}},n.xv=function(t,A,e,r,P,i,D){var u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt,Gt,dt,Bt,b1,f1,St,gt,g1,q1,U1,G1,y1,o1;for(t.k=A,n._v(t.k),t.s=new s.rub(),t.a=new s.tsb(),t.r&&h.yhb(t.r,0),t.b=0,t.p=false,l=t.q!=0,Gt=null,b=B(Q.S7,H,6,32,15,1),b[0]=-1,dt=B(Q.S7,H,6,16,15,1),f1=B(Q.S7,H,6,16,15,1),b1=B(Q.S7,H,6,16,15,1),Bt=B(Q.S7,H,6,16,15,1),ut=0;ut<16;ut++)dt[ut]=-1;for(T=0,V=-1,St=false,st=false,K=false,R=0,F=1,k=0;e[r]<=32;)++r;for(;r<P;){if(gt=e[r++]&n1,h.cib(gt)||gt==42||gt==63||gt==33&&l&&St||gt==35&&l&&St||gt==36&&l&&St){if(c=new n.iv(t,t.j|t.q),St){if((t.j&128)!=0){if(x=new n.Lv(r-1),r=n.dv(c,e,r,P,true,true),e[r-1]!=93){for(;e[r-1]!=93;)r=n.dv(c,e,r+1,P,true,true),++x.a;t.f.add(x);}}else r=n.dv(c,e,r,P,l,false);}else r=n.fv(c,e,r,P,l);if(St=false,c.q&&s.xj(c.q,0)){V=b[R],b[R]=A.q,n.Yk(A,c.q?s.xj(c.q,0):null),V!=-1&&F!=512&&(g=n.Wk(t.k,V,V,F),k!=0&&(t.p=true,n.Ln(t.k,g,k,true),n.Sk(t.k,g))),F=1,k=0;continue;}if(c.d==-1&&gt!=63)throw Lt(new h.cC("SmilesParser: unknown element label found. Position:"+(r-1)));c.r&&(t.p=true),y=n.av(c,t.k,gt,r),n.Km(t.k,y)&&++t.b,V=b[R],V!=-1&&F!=512&&(g=n.Wk(t.k,V,y,F),k!=0&&(t.p=true,n.Ln(t.k,g,k,true),n.Sk(t.k,g))),F=1,k=0,b[R]=y,T!=0&&(n.tn(t.k,y,T),T=0),D&&(Jt=Gt?s.gqb(Gt,h.sjb(V)):null,Jt&&n.Nv(Jt,y,(wt=r,c.d==1&&n.Ml(t.k,y)==0,wt)),c.p&&(!Gt&&(Gt=new s.Zwb()),it=c.g==9?0:c.g,s.Rwb(Gt,h.sjb(y),new n.Tv(t,y,r-2,V,it,c.i))));continue;}if(gt==46){b[R]=-1,F=512;continue;}if(gt==45||gt==61||gt==35||gt==36||gt==58||gt==47||gt==92||gt==60||gt==126||gt==33||gt==64){if(St)throw Lt(new h.cC("SmilesParser: unexpected bond symbol inside square brackets: '"+String.fromCharCode(gt)+TP+(r-1)));for(G=0;gt==45||gt==61||gt==35||gt==36||gt==58||gt==47||gt==92||gt==60||gt==126||gt==33||gt==64;){if(gt==33){if(gt=e[r++]&n1,gt==64)k|=128;else if(gt==45&&e[r]==62||gt==60&&e[r]==45)G|=16,++r;else if(gt==45)G|=1;else if(gt==61)G|=2;else if(gt==35)G|=4;else if(gt==36)G|=32;else if(gt==58)G|=8;else throw Lt(new h.cC("SmilesParser: bond symbol '"+String.fromCharCode(gt)+"' not allowed after '!'. Position:"+(r-1)));}else if(gt==64?k|=256:gt==61?F=2:gt==35?F=4:gt==36?F=32:gt==58?F=8:gt==126?k|=15:gt==47?D&&(F=257):gt==92?D&&(F=129):(gt==45&&e[r]==62||gt==60&&e[r]==45)&&(F=16,++r),e[r]==44)for(k|=(F==16?62:gt)==61?2:(F==16?62:gt)==35?4:(F==16?62:gt)==36?32:(F==16?62:gt)==58?8:(F==16?62:gt)==62?16:(F==16?62:gt)==126?127:1;e[r]==44;)e[r+1]==60&&e[r+2]==45||e[r+1]==45&&e[r+2]==62?(k|=16,r+=3):(k|=n.sv(e[r+1]&n1),r+=2);if(e[r]==59){++r,gt=e[r++]&n1;continue;}G!=0&&(k|=127&~G);break;}continue;}if(gt<=32){r=P;continue;}if(AA(gt)){if(Vt=gt-48,St){for(;r<P&&AA(e[r]&n1);)Vt=10*Vt+e[r]-48,++r;T=Vt;}else {if(S=st?r-3:r-2,ot=e[S]==45||e[S]==47||e[S]==92||e[S]==61||e[S]==35||e[S]==36||e[S]==58||e[S]==62||e[S]==126,st&&r<P&&AA(e[r]&n1)&&(Vt=10*Vt+e[r]-48,st=false,++r),Vt>=dt.length){if(Vt>=100)throw Lt(new h.cC("SmilesParser: ringClosureAtom number out of range: "+Vt));for(Et=dt.length,ct=dt.length;ct<=Vt;)ct=d.Math.min(100,ct+16);for(dt=(w.EAb(ct),s.Drb(dt,B(Q.S7,H,6,ct,15,1),0,ct)),f1=(w.EAb(ct),s.Drb(f1,B(Q.S7,H,6,ct,15,1),0,ct)),b1=(w.EAb(ct),s.Drb(b1,B(Q.S7,H,6,ct,15,1),0,ct)),Bt=(w.EAb(ct),s.Drb(Bt,B(Q.S7,H,6,ct,15,1),0,ct)),Dt=Et;Dt<ct;Dt++)dt[Dt]=-1;}if(dt[Vt]==-1)dt[Vt]=b[R],f1[Vt]=r-1,b1[Vt]=ot?F:-1,Bt[Vt]=ot?k:0;else {if(dt[Vt]==b[R])throw Lt(new h.cC("SmilesParser: ring closure to same atom"));D&&Gt&&(Jt=s.gqb(Gt,h.sjb(dt[Vt])),Jt&&n.Nv(Jt,b[R],f1[Vt]),Jt=s.gqb(Gt,h.sjb(b[R])),Jt&&n.Nv(Jt,dt[Vt],r-1)),b1[Vt]!=-1?F=b1[Vt]:F==257?F=129:F==129&&(F=257),u=dt[Vt],o=b[R],g=n.pv(t,u,o,F,f1[Vt],r-1),Bt[Vt]!=0&&(k=Bt[Vt]),k!=0&&(t.p=true,n.Ln(t.k,g,Bt[Vt],true),n.Sk(t.k,g)),dt[Vt]=-1;}F=1,k=0;}continue;}if(gt==43)throw Lt(new h.cC("SmilesParser: '+' found outside brackets. Position:"+(r-1)));if(gt==40){if(b[R]==-1){K=true;continue;}++R,b.length==R&&(b=s.Brb(b,b.length+32)),b[R]=b[R-1];continue;}if(gt==41){if(R==0){if(!K)throw Lt(new h.cC("SmilesParser: Closing ')' without opening counterpart. Position:"+(r-1)));b[0]=-1,K=false;continue;}--R;continue;}if(gt==91){St=true;continue;}if(gt==93)throw Lt(new h.cC("SmilesParser: closing bracket at unexpected position:"+(r-1)));if(gt==37){st=true;continue;}throw Lt(new h.cC("SmilesParser: unexpected character outside brackets: '"+String.fromCharCode(gt)+ti+(r-1)));}if(F!=1)throw Lt(new h.cC("SmilesParser: dangling open bond; position:"+r));for(Qt=0;Qt<dt.length;Qt++)if(dt[Qt]!=-1)throw Lt(new h.cC("SmilesParser: dangling ring closure: "+Qt+"; position:"+r));for(At=n.lp(t.k),n.Qn(t.k,true),n.bw(t.k,1),U=0;U<t.k.q;U++)if(n.El(t.k,U)!=null){if(J=n.Fl(t.k,U)[0],t.p||t.q==2){if(J!=9)if(t.i)for(lt=0;lt<J;lt++)n.Wk(t.k,U,n.Uk(t.k,1),1);else J==0&&n.vn(t.k,U,1792,true),J==1&&n.vn(t.k,U,1664,true),J==2&&n.vn(t.k,U,1408,true),J==3&&n.vn(t.k,U,896,true);}else {if(J==9&&(J=0),!n.Lm(t.k,U)&&(!n.Km(t.k,U)||n.Tl(t.k,U)==6&&n.Bl(t.k,U)==0)){for(o1=n.mo(n.Tl(t.k,U)),E=false,g1=n.vp(t.k,U),g1-=n.mm(t.k,U,g1),g1+=J,n.Km(t.k,U)&&++g1,U1=o1,G1=0,y1=U1.length;G1<y1;++G1)if(q1=U1[G1],g1<=q1){E=true,q1==g1+2?n.wn(t.k,U,48):q1==g1+1?n.wn(t.k,U,32):(q1!=g1||q1!=o1[0])&&n.gn(t.k,U,g1);break;}E||n.gn(t.k,U,g1);}if(t.i||!n.Aq(t.k,U))for(lt=0;lt<J;lt++)n.Wk(t.k,U,n.Uk(t.k,1),1);}}else !t.i&&(t.p||t.q==2)&&(J=n.cp(t.k,U),J>=1&&n.vn(t.k,U,128,true),J>=2&&n.vn(t.k,U,256,true),J>=3&&n.vn(t.k,U,512,true),J>=4&&n.vn(t.k,U,f0,true));if(!t.i&&(t.p||t.q==2)&&n.qq(t.k,false),n.bw(t.k,1),n.uv(t),n.wv(t,l,t.p),t.k.t=null,n.Qn(t.k,false),D&&(n.rv(t),Gt)){for(Nt=(Ht=new s.Gpb(Gt).a.cd().Tb(),new s.Npb(Ht));Nt.a.ed();)Jt=(O=Nt.a.fd(),O.jd()),n.un(t.k,At[Jt.a],n.Ov(Jt,At),false);n.vq(t.k,0);}n.vq(t.k,0),i&&(bt=new X.CA(t.d),Rt(t.n,0)&&X.yA(bt,t.n),X.jA(bt,t.k),D&&n.ow(t.k)),(t.p||t.q==2)&&(n.On(t.k,true),n.Cq(t.k),n.Dq(t.k));},n.yv=function(t,A,e,r,P){n.xv(t,A,e,0,e.length,r,P);},n.zv=function(t,A){var e;return A==null?null:n.Av(t,w.uAb((e=A,O1.Enb(),e)));},n.Av=function(t,A){var e,r,P,i,D,u,o,l,U,y;for(P=Ut.p4(A,0);P>0&&A[P-1]==45;)P=Ut.p4(A,P+1);for(i=P==-1?-1:Ut.p4(A,P+1);i>0&&A[i-1]==45;)i=Ut.p4(A,i+1);if(i==-1)throw Lt(new h.cC("Missing one or both separators ('>')."));if(Ut.p4(A,i+1)!=-1)throw Lt(new h.cC("Found more than 2 separators ('>')."));for(y=new tt.zH(),U=0,r=0;r<A.length;){for(;r<A.length&&A[r]==46;)++r;if(r==A.length)break;for(l=r,o=-1,D=0,e=r;e<A.length;e++){if(A[e]==40){++D;continue;}if(A[e]==41){if(--D<0)throw Lt(new h.cC("Found closing bracket without matching opening one."));continue;}if(D==0&&(A[e]==46&&(t.o||e+1<A.length&&A[e+1]==46)||A[e]==62&&(e==0||A[e-1]!=45))){o=e;break;}}if(o==-1&&(o=A.length),l<o){if(u=new n.tw(),n.xv(t,u,A,l,o,true,true),t.q==1&&t.p)return n.Av(new n.Jv(t.j|2),A);U==0?(s.sj(y.g,u),y.d=-1):U==1?s.sj(y.a,u):(s.sj(y.f,u),y.d=-1);}for(r=o;r<A.length&&A[r]==62;)++r,++U;}return y;},n.Bv=function(t,A){var e,r,P,i;for(n.im(t.k,A)==1&&n.Mn(t.k,A,2),P=0;P<2;P++)for(e=n.Zl(t.k,P,A),n.Km(t.k,e)&&(n.sn(t.k,e,false),--t.b),i=0;i<n.$o(t.k,e);i++)r=n._o(t.k,e,i),t.g[r]&&(t.g[r]=false,--t.c);},n.Cv=function(t){var A,e,r,P,i,D,u;do for(u=false,e=0;e<t.k.g;e++)if(t.g[e]){for(i=false,P=0;P<2;P++){for(A=false,r=n.Zl(t.k,P,e),D=0;D<n.$o(t.k,r);D++)if(e!=n._o(t.k,r,D)&&t.g[n._o(t.k,r,D)]){A=true;break;}if(!A){i=true;break;}}i&&(u=true,n.Bv(t,e));}while(u);},n.Dv=function(t){var A,e,r,P,i,D;for(A=0;A<t.k.g;A++)if(t.g[A]){for(i=true,P=0;P<2&&i;P++)for(e=n.Zl(t.k,P,A),D=0;D<n.$o(t.k,e)&&i;D++)A!=n._o(t.k,e,D)&&t.g[n._o(t.k,e,D)]&&(i=false);if(i)for(n.Mn(t.k,A,8),--t.c,r=0;r<2;r++)e=n.Zl(t.k,r,A),n.Km(t.k,e)&&(n.sn(t.k,e,false),--t.b);}},n.Ev=function(t,A){var e,r;return !n.Ft(n.Tl(t.k,A))||n.Tl(t.k,A)==6&&(!n.Km(t.k,A)||n.Bl(t.k,A)>0)||(e=n.El(t.k,A)==null||n.Fl(t.k,A)[0]==9?0:n.Fl(t.k,A)[0],r=n.jp(t.k,A)-e,r<1)?false:n.Tl(t.k,A)==16||n.Tl(t.k,A)==34||n.Tl(t.k,A)==52?n.$o(t.k,A)==2&&n.Bl(t.k,A)<=0?false:r!=2:true;},n.Fv=function(t,A){t.f=A;},n.Gv=function(t,A){t.n=A;},n.Hv=function(t,A){t.e&&(!t.r&&(t.r=new h.Mkb()),h.Kkb(t.r," "),h.Kkb(t.r,A));},n.Iv=function(){n.Jv.call(this,0);},n.Jv=function(t){this.j=t&-4,this.q=t&3,this.o=(t&32)!=0,this.e=(t&64)!=0,this.i=(t&8)!=0,this.d=2,(t&4)!=0&&(this.d|=1),this.i&&(this.d&=-3);},I(113,1,{},n.Iv,n.Jv),L.b=0,L.c=0,L.d=0,L.e=false,L.i=false,L.j=0,L.n=0,L.o=false,L.p=false,L.q=0,Q.Q8=N(113),n.Kv=function(t,A){return P8(A.b,t.b);},n.Lv=function(t){this.b=t,this.a=1;},I(216,1,{216:1,32:1},n.Lv),L.Sb=function(t){return n.Kv(this,t);},L.a=0,L.b=0,Q.N8=N(216),n.Nv=function(t,A,e){if(!t.c){if(t.e.a.length==4){t.c=true;return;}s.sj(t.e,new n.Uv(t,A,e));}},n.Ov=function(t,A){var e,r,P;if(t.c)return 3;for(P=new s.trb(t.e);P.a<P.c.a.length;)r=s.srb(P),r.a!=Le&&r.a!=Ot&&(r.a=A[r.a]);switch(e=false,t.e.a.length){case 2:if(s.xj(t.e,0).a>=Le||s.xj(t.e,1).a>=Le)return 3;e=n.Rv(t,A);break;case 3:s.sj(t.e,new n.Uv(t,Ot,t.b));case 4:e=n.Sv(t);break;default:return 3;}return t.d^e?1:2;},n.Pv=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it;if(R=t.f.k,P=s.xj(t.e,e),it=P.a,y=A[t.a],k=n.op(R,it)>0&&R.j[it]==R.e[it],g=s.psb(t.f.a,t.f.t[it]),!k&&!g)return D=R.i[it][2],R.C[D]==1&&(D=t.f.t[D],(e==0?D>it:t.f.t[R.i[it][1]]>D)&&(r=!r)),r;if(k&&(i=R.i[it][e],D=t.f.t[i],D>it&&(r=!r,R.C[i]==1&&(r=!r))),g){for(E=R.e[it],c=B(Q.S7,H,6,E-1,15,1),b=B(Q.S7,H,6,E-1,15,1),O=0,F=0,G=0;F<E;F++)D=R.i[it][F],D!=y&&(c[G]=D,l=n.Xo(R,it,D),At=s.qob(t.f.s,t.f.t[it]+"_"+t.f.t[D]),b[G++]=At==null?-1:l,At!=null&&++O);if(k)c[0]<it&&(r=!r);else switch(O){case 1:if(n.op(R,it)>0)break;S=b[0]>=0,T=c[S?0:1],x=c[S?1:0],x>it?T>it&&R.C[x]!=1&&(r=!r):T<it&&(r=!r);break;case 2:u=n.Qv(t,b[0],it),o=n.Qv(t,b[1],it),ot=s.qob(t.f.s,it+"_"+t.f.t[u]),K=s.qob(t.f.s,it+"_"+t.f.t[o]),U=t.b,e==0?(J=ot[1]<U?ot[1]:ot[0],V=K[1]<U?K[1]:K[0]):(J=ot[0]<U?ot[1]:ot[0],V=K[0]<U?K[1]:K[0]),J<V!=u<o&&(r=!r);}}return r;},n.Qv=function(t,A,e){return t.f.k.D[0][A]==e?t.f.k.D[1][A]:t.f.k.D[0][A];},n.Rv=function(t,A){var e,r;if(r=false,t.f.t==null)for(t.f.t=B(Q.S7,H,6,A.length,15,1),e=t.f.t.length;--e>=0;)t.f.t[A[e]]=e;return r=n.Pv(t,A,0,r),r=n.Pv(t,A,1,r),r;},n.Sv=function(t){var A,e,r;for(e=false,A=1;A<t.e.a.length;A++)for(r=0;r<A;r++)s.xj(t.e,r).a>s.xj(t.e,A).a&&(e=!e),s.xj(t.e,r).b>s.xj(t.e,A).b&&(e=!e);return e;},n.Tv=function(t,A,e,r,P,i){this.f=t,P!=0&&P!=1?this.c=true:(this.a=A,this.b=e,this.d=i,this.e=new s.Mj(),r!=-1&&n.Nv(this,r,e-1),r!=-1&&P==1&&n.Nv(this,Le,e+1));},I(217,1,{217:1},n.Tv),L.a=0,L.b=0,L.c=false,L.d=false,Q.P8=N(217),n.Uv=function(t,A,e){this.c=t,this.a=A,this.b=e;},I(182,1,{182:1},n.Uv),L.pb=function(){return "["+(this.a==Le?"h":this.a==Ot?"lp":n.Il(this.c.f.k,this.a))+this.b+"]";},L.a=0,L.b=0,Q.O8=N(182),n.Wv=function(t){t.c=new s.Mj();},n.Xv=function(t,A){var e,r,P,i;if(i=t.c.a.length,i==0)return s.rj(t.c,0,A),0;for(P=1;2*P<=i;)P<<=1;for(r=P,--P;r!=0;){if(r>>=1,P>=i){P-=r;continue;}if(e=h.Yjb(A,s.xj(t.c,P)),e==0)return  -1;if(r==0)break;e<0?P-=r:P+=r;}return P<i&&h.Yjb(A,s.xj(t.c,P))>0&&++P,s.rj(t.c,P,A),P;},n.Yv=function(t,A){var e,r,P,i;if(i=t.c.a.length,i==0)return  -1;for(P=1;2*P<=i;)P<<=1;for(r=P,--P;r!=0;){if(r>>=1,P>=i){P-=r;continue;}if(e=h.Yjb(A,s.xj(t.c,P)),e==0)return P;if(r==0)break;e<0?P-=r:P+=r;}return  -1;},n.Zv=function(){n.Wv(this);},I(102,1,ii,n.Zv),L.bc=function(t){return n.Yv(this,t);},Q.R8=N(102),n._v=function(t){n.jl(t),t.d=null;},n.aw=function(t,A){n.pl(t,A),t.d&&(A.T=0);},n.bw=function(t,A){var e,r,P,i;if(n.Eo(t,A),(A&~t.T)!=0){for((t.T&15)!=0&&t.q>1&&n.ei(t.J[0],t.J[1])&&X.jA(new X.CA(0),t),t.c&&(A|=128),e=0;e<t.q;e++)t.u[e]&=-67223560;for(r=0;r<t.g;r++)t.F[r]&=-64;P=0,i=0,(A&32)!=0?(P=32,i=1):(A&64)!=0&&(P=64,i=3),(A&128)!=0&&(P|=128,i|=32),t.d=new n.Og(t,i),n.Jg(t.d),n.Kg(t.d),n.Ig(t.d),n.sw(t)&&(t.d=new n.Og(t,i)),t.T|=24|P;}},n.cw=function(t,A){return n.wg(t.d,A);},n.dw=function(t,A){return n.lg(t.d,A);},n.ew=function(t){var A;switch(n.bw(t,31),A=t.I&n1,t.I&fA){case H1:return null;case r0:return A==1?"meso":""+A+" meso diastereomers";case 0:return "unknown chirality";case Ye:return "both enantiomers";case yA:return "this enantiomer";case 327680:return "this or other enantiomer";case tn:return "two epimers";default:return A==1?"one stereo isomer":""+A+" stereo isomers";}},n.fw=function(t){var A;return A=new n.uw(t.q,t.r),n.ol(t,A),A;},n.gw=function(t){var A,e,r,P,i,D,u;for(n.bw(t,15),D=B(Q.S7,H,6,3,15,1),e=0;e<t.f;e++)(t.u[e]&O0)!=0&&(u=(t.u[e]&l1)>>18,u!=0&&(D[u]=d.Math.max(D[u],(t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&v0)>>20)));for(P=0;P<t.g;P++)((t.F[P]&3)==1||(t.F[P]&3)==2)&&t.H[P]==1&&(u=(t.F[P]&768)>>8,u!=0&&(D[u]=d.Math.max(D[u],(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10)));for(i=B(Q.S7,E1,7,3,0,2),i[1]=B(Q.S7,H,6,1+D[1],15,1),i[2]=B(Q.S7,H,6,1+D[2],15,1),A=0;A<t.f;A++)(t.u[A]&O0)!=0&&(u=(t.u[A]&l1)>>18,u!=0&&++i[u][(t.u[A]&l1)>>18!=1&&(t.u[A]&l1)>>18!=2?-1:(t.u[A]&v0)>>20]);for(r=0;r<t.g;r++)((t.F[r]&3)==1||(t.F[r]&3)==2)&&t.H[r]==1&&(u=(t.F[r]&768)>>8,u!=0&&++i[u][(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10]);return i;},n.hw=function(t){var A,e;return A=B(Q.S7,H,6,t.q,15,1),e=n.hp(t,A,false,false),n.iw(t,A,e);},n.iw=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F;for(k=B(Q.S8,GA,26,e,0,1),D=B(Q.S7,H,6,e,15,1),l=B(Q.S7,H,6,e,15,1),i=B(Q.S7,H,6,t.q,15,1),P=0;P<t.q;P++)A[P]!=-1&&(i[P]=D[A[P]]++);for(o=0;o<t.r;o++)b=A[t.D[0][o]],g=A[t.D[1][o]],b==g&&b!=-1&&++l[b];for(F=0;F<e;F++)k[F]=new n.uw(D[F],l[F]),n.aw(t,k[F]);for(r=0;r<t.q;r++)A[r]!=-1&&n.ll(t,k[A[r]],r,0,0);for(u=0;u<t.r;u++)b=A[t.D[0][u]],g=A[t.D[1][u]],b==g&&b!=-1&&n.nl(t,k[b],u,0,0,i,false);for(y=k,T=0,c=y.length;T<c;++T)U=y[T],n.cn(U,1),n.cn(U,2);return k;},n.jw=function(t){return n.bw(t,15),!t.d&&(t.f<2||!n.ei(t.J[0],t.J[1]))&&(t.d=new n.Ng(t)),t.d?n.rg(t.d):null;},n.kw=function(t){return n.bw(t,15),!t.d&&(t.f<2||!n.ei(t.J[0],t.J[1]))&&(t.d=new n.Ng(t)),t.d?n.mg(t.d):null;},n.lw=function(t){var A,e;for(n.bw(t,31),e=0,A=0;A<t.f;A++)(t.u[A]&3)!=0&&(t.u[A]&4)==0&&++e;return e;},n.mw=function(t,A){return n.vg(t.d,A);},n.nw=function(t,A){t.c=A,t.T&=-144;},n.ow=function(t){n.bw(t,31),t.d&&n.Lg(t.d);},n.pw=function(t){var A,e;for(n.bw(t,15),t.L=false,A=0;A<t.q;A++)t.u[A]&=Ar,(t.u[A]&3)!=0?t.u[A]|=Qe:t.u[A]&=tr;for(e=0;e<t.r;e++)(t.F[e]&3)!=0&&n.fm(t,e)==2?t.H[e]=386:t.H[e]&=-385;t.T&=-249;},n.qw=function(t,A,e){var r,P,i,D,u,o,l,U,y;if(y=t.u[A]&3,y==1||y==2){if(u=false,t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2&&t.C[A]<=7)for(D=0;D<t.j[A];D++){for(r=t.i[A][D],U=0,l=B(Q.S7,H,6,3,15,1),o=0;o<t.j[r];o++)l[U]=t.i[r][o],l[U]!=A&&++U;U==2&&l[0]<l[1]^e[l[0]]<e[l[1]]&&(u=!u);}else for(D=1;D<t.j[A];D++)for(o=0;o<D;o++)P=t.i[A][D],i=t.i[A][o],P<i^e[P]<e[i]&&(u=!u);u&&(y=y==1?2:1);}return y;},n.rw=function(t){var A,e,r,P,i,D;for(n.Bq(t),n.bw(t,31),r=0;r<t.f;r++){if(((t.u[r]&l1)>>18==1||(t.u[r]&l1)>>18==2)&&((t.u[r]&O0)==0||(t.u[r]&3)==3))throw Lt(new h.cC(hP));if((t.u[r]&H1)!=0)throw Lt(new h.cC(cP));if(((t.u[r]&3)==1||(t.u[r]&3)==2)&&t.o[r]==0){for(A=B(Q.Q7,jt,6,t.j[r],15,1),i=0;i<t.j[r];i++)A[i]=n.Yl(t,r,t.i[r][i]);for(P=1;P<t.j[r];P++)if(!n.Sm(t,t.k[r][P],r)){for(D=0;D<P;D++)if(!n.Sm(t,t.k[r][D],r)&&(e=d.Math.abs(n.oo(A[P],A[D])),e<.08||e>z8))throw Lt(new h.cC(bP));}}}},n.sw=function(t){var A,e,r,P,i,D,u;for(u=false,e=0;e<t.f;e++)((t.u[e]&O0)==0||(t.u[e]&3)==3)&&(t.u[e]&=Ar);for(i=0;i<t.g;i++)(n.fm(t,i)!=1||(t.F[i]&3)==0||(t.F[i]&3)==3)&&(t.F[i]&=-32513);if(t.L){if((t.I&fA)!=r0){for(D=B(Q.xfb,Tt,6,t.f,16,1),r=0;r<t.f;r++)(t.u[r]&O0)!=0&&(t.u[r]&3)!=3&&(t.u[r]&l1)>>18==1&&(D[r]=true);for(P=0;P<t.f;P++)(t.u[P]&O0)!=0&&(t.u[P]&3)!=3&&(n.on(t,P,1,0),u=true);for(A=0;A<t.f;A++)D[A]&&(n.un(t,A,1,false),n.on(t,A,1,-1),u=true);}t.L=false;}return n.cn(t,1),n.cn(t,2),u;},n.tw=function(){n.Rk(),n.Fq.call(this);},n.uw=function(t,A){n.Rk(),n.Gq.call(this,t,A);},n.vw=function(t){n.Rk(),n.Hq.call(this,t);},I(26,114,fi,n.tw,n.uw,n.vw),L.Yb=function(t){n.aw(this,t);},L.$b=function(t){n.bw(this,t);},L.c=false,Q.S8=N(26),n.zw=function(t,A){var e,r,P,i,D,u,o,l,U,y,T;for(u=new s.Mj(),n.bw(A,1),l=B(Q.xfb,Tt,6,A.f,16,1),e=0;e<A.f;e++)if(t.f[e]&&(n.Tl(t.i,e)==7||n.Tl(t.i,e)==8||n.Tl(t.i,e)==16||n.Tl(t.i,e)==34||n.Tl(t.i,e)==52)){for(l[e]=true,o=0;o<A.j[e];o++)if(r=A.i[e][o],i=A.k[e][o],y=A.n[e][o],A.o[r]!=0&&A.o[e]<y){for(l[r]=true,s.sj(u,h.sjb(i)),U=0;U<A.j[r];U++)P=A.i[r][U],l[P]||(l[P]=true,D=A.k[r][U],T=A.n[r][U],A.o[r]+2==y+T&&(s.sj(u,h.sjb(D)),y>=T&&(A.o[P]<T?n.Iw(A,P)&&n.Cw(t,A,P,true,false,l,u):n.Cw(t,A,P,true,true,l,u)),T>=y&&n.Iw(A,e)&&n.Cw(t,A,P,false,false,l,u),t.f[P]&&(n.Tl(t.i,P)==5||n.Tl(t.i,P)==6||n.Tl(t.i,P)==7||n.Tl(t.i,P)==8||n.Tl(t.i,P)==16||n.Tl(t.i,P)==34||n.Tl(t.i,P)==52)&&A.o[P]<T&&(y<=2&&T>=2&&n.Iw(A,e)&&n.Aw(t,A,i,D),y>=2&&T<=2&&n.Iw(A,P)&&n.Aw(t,A,D,i)),s.Ej(u,u.a.length-1)),l[P]=false);s.Ej(u,u.a.length-1),l[r]=false;}l[e]=false;}},n.Aw=function(t,A,e,r){var P;P=new n.Ow(t,A),n.Nw(P,e,n.fm(A,e)==1?2:3),n.Nw(P,r,n.fm(A,r)==2?1:2),t.g[e]=true,t.g[r]=true,s.Kxb(t.e,P)&&s.Mqb(t.d,P);},n.Bw=function(t,A){s.Kxb(t.e,A)&&s.Mqb(t.d,A);},n.Cw=function(t,A,e,r,P,i,D){var u,o,l,U,y,T,c,b,g,k,F,S;for(c=0;c<A.j[e];c++)if(u=A.i[e][c],!i[u]&&(U=A.k[e][c],F=n.fm(A,U),r&&F>=2||!r&&F<=2)){for(i[u]=true,D.add(h.sjb(U)),b=0;b<A.j[u];b++)if(o=A.i[u][b],!i[o]&&(y=A.k[u][b],S=n.fm(A,y),A.o[u]+2==F+S&&(r&&S<=2||!r&&S>=2))){if(i[o]=true,D.add(h.sjb(y)),t.f[o]&&(n.Tl(t.i,o)==5||n.Tl(t.i,o)==6||n.Tl(t.i,o)==7||n.Tl(t.i,o)==8||n.Tl(t.i,o)==16||n.Tl(t.i,o)==34||n.Tl(t.i,o)==52)&&(!r||n.Iw(A,o))){for(T=new n.Ow(t,A),g=0;g<D.a.length;g++)l=(w.HAb(g,D.a.length),D.a[g]).a,k=g<2?r^(g&1)==0:P^(g&1)==0,k?n.Nw(T,l,n.fm(A,l)==1?2:3):n.Nw(T,l,n.fm(A,l)==2?1:2),t.g[l]=true;s.Kxb(t.e,T)&&s.Mqb(t.d,T);}else n.Cw(t,A,o,r,P,i,D);s.Ej(D,D.a.length-1),i[o]=false;}s.Ej(D,D.a.length-1),i[u]=false;}return  false;},n.Dw=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(t.b=B(Q.S7,H,6,t.i.f,15,1),u=B(Q.S7,H,6,t.i.f,15,1),r=B(Q.xfb,Tt,6,t.i.g,16,1),y=0,e=0;e<t.i.g;e++)if(!r[e]&&t.g[e])for(++y,t.b[n.Zl(t.i,0,e)]=y,t.b[n.Zl(t.i,1,e)]=y,r[e]=true,l=0;l<2;l++)for(A=n.Zl(t.i,l,e),t.b[A]=y,D=0,o=0,u[0]=A;D<=o;){for(U=0;U<n.$o(t.i,u[D]);U++)i=n._o(t.i,u[D],U),!r[i]&&t.g[i]&&(r[i]=true,P=n.Zo(t.i,u[D],U),t.b[P]==0&&(t.b[P]=y,u[++o]=P));++D;}t.j=y;},n.Ew=function(t){var A,e,r;for(t.n=B(Q.S7,H,6,t.j,15,1),t.k=B(Q.S7,H,6,t.j,15,1),t.o=B(Q.S7,H,6,t.j,15,1),A=0;A<t.i.f;A++)t.b[A]!=0&&(r=t.b[A]-1,t.a!=null&&(t.k[r]+=t.a[A]),t.c!=null&&(t.o[r]+=t.c[A]));for(e=0;e<t.i.g;e++)t.g[e]&&n.fm(t.i,e)==2&&(t.n[t.b[n.Zl(t.i,0,e)]-1]+=2);},n.Fw=function(t){var A,e,r,P;for(r=0;r<t.i.r;r++)for(P=0;P<2;P++)A=n.Zl(t.i,P,r),e=n.Zl(t.i,1-P,r),n.Tl(t.i,A)==1&&n.Ml(t.i,A)>1&&n.Tl(t.i,e)>1&&t.b[e]!=0&&(n.Ml(t.i,A)==2?(t.a==null&&(t.a=B(Q.S7,H,6,t.i.q,15,1)),++t.a[e]):(t.c==null&&(t.c=B(Q.S7,H,6,t.i.q,15,1)),++t.c[e]),n.Um(t.i,A));(t.a!=null||t.c!=null)&&n.vl(t.i);},n.Gw=function(t){var A;for(t.e=new s.Pxb(),t.d=new s.Yqb(),n.Bw(t,new n.Ow(t,t.i)),A=n.fw(t.i);!s.Qqb(t.d);)if(n.Mw(s.Sqb(t.d),A),n.zw(t,A),t.e.a.c>=n.yw){h.Rkb(),""+n.rg(new n.Ng(t.i));break;}},n.Hw=function(t){var A,e,r,P,i,D,u,o,l,U;if(t.e.a.c==1)return t.i;for(l=n.fw(t.i),n.On(l,true),n.bw(l,7),P=0;P<l.g;P++)t.g[P]&&(l.H[P]=1,l.T=0,n.Ln(l,P,3,true));for(e=0;e<l.f;e++)t.b[e]!=0&&n.tp(t.i,e)<4&&(n.Bo(l,e),l.u[e]&=tr,l.T&=7,n.on(l,e,0,-1));for(u=B(Q.S7,H,6,t.j,15,1),o=B(Q.S7,H,6,t.j,15,1),r=new n.Ng(l).f,A=0;A<l.f;A++)t.b[A]!=0&&(U=t.b[A]-1,o[U]<r[A]&&(o[U]=r[A],u[U]=A));for(i=0;i<t.j;i++)D=""+t.n[i]+"|"+t.k[i]+"|"+t.o[i],n.mn(l,u[i],D);return l;},n.Iw=function(t,A){return t.e[A]-n.tp(t,A)+n.op(t,A)<=0?false:n.Hm(t,A)?true:t.o[A]==0;},n.Jw=function(t){var A,e,r,P,i,D,u,o;n.bw(t.i,1),u=t.i.f;do--u;while(u>=0&&n.Tl(t.i,u)==1);for(A=0;A<u;A++)if(n.Tl(t.i,A)==1){n.Zn(t.i,A,u);do--u;while(n.Tl(t.i,u)==1);}if(u!=t.i.f-1){for(D=B(Q.xfb,Tt,6,t.i.g,16,1),i=0;i<t.i.g;i++)e=n.Zl(t.i,0,i),r=n.Zl(t.i,1,i),(n.Tl(t.i,e)==1||n.Tl(t.i,r)==1)&&(D[i]=true);o=t.i.g;do--o;while(o>=0&&D[o]);for(P=0;P<o;P++)if(D[P]){n.$n(t.i,P,o),D[P]=false;do--o;while(D[o]);}}},n.Kw=function(t){var A,e;for(this.i=n.fw(t),n.Jw(this),n.bw(this.i,7),this.g=B(Q.xfb,Tt,6,this.i.g,16,1),this.f=B(Q.xfb,Tt,6,this.i.f,16,1),A=0;A<this.i.f;A++)e=n.mo(n.Tl(this.i,A))[0],this.f[A]=n.tp(this.i,A)<e;n.Gw(this),n.Dw(this),n.Fw(this),n.Ew(this);},I(241,1,{},n.Kw),L.j=0,n.yw=ur,Q.U8=N(241),n.Lw=function(t,A){return Ut.H4(t.a,A.a);},n.Mw=function(t,A){var e,r;for(r=0;r<t.b.i.g;r++)t.b.g[r]&&(e=3&t.a[r>>4]>>2*(r&15),n.Mn(A,r,e==1?1:e==2?t.b.g[r]&&!n.jq(t.b.i,r)?386:2:e==3?4:16));},n.Nw=function(t,A,e){var r,P;r=A>>4,P=2*(A&15),t.a[r]&=~(3<<P),t.a[r]|=e<<P;},n.Ow=function(t,A){var e;for(this.b=t,this.a=B(Q.S7,H,6,(t.i.g+15)/16|0,15,1),e=0;e<t.i.g;e++)this.a[e>>4]|=d.Math.min(3,n.fm(A,e))<<2*(e&15);},I(166,1,{166:1,32:1},n.Ow),L.Sb=function(t){return n.Lw(this,t);},Q.T8=N(166),n.Qw=function(t,A,e){var r,P,i,D,u;for(P=0,D=new s.trb(e);D.a<D.c.a.length;)i=s.srb(D),w.UAb(i).length!=0&&(u=C.z3(A,i).b,P<u&&(P=u));return r=t.c*se*(e.a.length-1)+t.c,new ht.qO(t.i[0].a,t.i[0].b-t.c/2,P,r);},n.Rw=function(t,A,e){return !!t.b&&ht.lO(t.b,A,e);},n.Sw=function(t){var A,e,r,P,i;if(e=h.jkb(t,X1(38)),e==-1)return t;for(P=0,i=new h.Ekb();e!=-1;)h.Dkb(i,(w.OAb(P,e,w.UAb(t).length),w.UAb(t).substr(P,e-P))),w.PAb(e+1,w.UAb(t).length),w.UAb(t).charCodeAt(e+1)==38?(i.a+="&",P=e+2):(A=(w.PAb(e+1,w.UAb(t).length),w.UAb(t).charCodeAt(e+1)),A=A-(A<65?48:A<97?65:97),r=(w.PAb(e+2,w.UAb(t).length),w.UAb(t).charCodeAt(e+2)),r=r-(r<65?48:r<97?65:97),h.Bkb(i,16*A+r&n1),P=e+3),e=h.kkb(t,X1(38),P);return h.Dkb(i,(w.PAb(P,w.UAb(t).length+1),w.UAb(t).substr(P))),i.a;},n.Tw=function(t){var A,e;for(e=new h.Ekb(),A=0;A<w.UAb(t).length;A++)switch(w.PAb(A,w.UAb(t).length),w.UAb(t).charCodeAt(A)){case 38:e.a+="&&";break;case 9:e.a+="&09";break;case 10:e.a+="&0A";break;case 32:e.a+="&20";break;default:h.Bkb(e,(w.PAb(A,w.UAb(t).length),w.UAb(t).charCodeAt(A)));}return e.a;},n.Uw=function(t,A){var e;return e=n.Vw(t),n.Qw(t,A,e);},n.Vw=function(t){var A,e,r;if(e=new s.Mj(),A=h.jkb(t.e,X1(10)),A==-1)s.sj(e,t.e);else {for(r=0;A!=-1;)s.sj(e,h.tkb(t.e,r,A)),r=A+1,A=h.ikb(t.e,10,r);s.sj(e,h.skb(t.e,r));}return e;},n.Ww=function(t,A,e){t.i[0].a=A,t.i[0].b=e;},n.Xw=function(t,A){try{t.c=I0(A);}catch(e){if(e=M1(e),!Zt(e,40))throw Lt(e);}},n.Yw=function(t,A){try{t.d=h.Oib(A,e1,Ot);}catch(e){if(e=M1(e),!Zt(e,40))throw Lt(e);}},n.Zw=function(t,A,e,r){t.e=A,t.c=e,t.d=r;},n.$w=function(t,A){try{t.i[0].a=I0(A);}catch(e){if(e=M1(e),!Zt(e,40))throw Lt(e);}},n._w=function(t,A){try{t.i[0].b=I0(A);}catch(e){if(e=M1(e),!Zt(e,40))throw Lt(e);}},n.ax=function(){n.cx.call(this,new ht.bO());},n.bx=function(t){var A,e,r,P;for(n.ax.call(this),A=0;A!=-1&&(e=w.UAb(t).indexOf('="',A),e!=-1);)r=(w.OAb(A+1,e,w.UAb(t).length),w.UAb(t).substr(A+1,e-(A+1))),A=w.UAb(t).indexOf('"',e+2),P=A==-1?(w.PAb(e+1,w.UAb(t).length+1),w.UAb(t).substr(e+1)):(w.OAb(e+1,A,w.UAb(t).length),w.UAb(t).substr(e+1,A-(e+1))),h.ckb(r,"text")?this.e=n.Sw(P):h.ckb(r,"x")?n.$w(this,P):h.ckb(r,"y")?n._w(this,P):h.ckb(r,"size")?n.Xw(this,P):h.ckb(r,"style")&&n.Yw(this,P);},n.cx=function(t){this.e="",this.c=9,this.d=0,this.i=B(Q.Zab,C1,55,1,0,1),this.i[0]=t;},I(94,193,{193:1,94:1},n.ax,n.bx),L.Eb=function(t,A){return this.a=!!this.b&&ht.lO(this.b,t,A),this.a;},L.Fb=function(){this.a=false;},L.Gb=function(){var t;return t=new n.ax(),n.Zw(t,this.e,this.c,this.d),n.Ww(t,this.i[0].a,this.i[0].b),t.g=this.g,t;},L.Hb=function(t,A){return n.Rw(this,t,A);},L.Ib=function(t,A){var e,r,P;for(r=A?A.c*this.c:this.c,C.F3(t,pt(d.Math.round(r)),(this.d&1)!=0,(this.d&2)!=0),C.H3(t,this.g?fA:C.E3(t)?-1:G0),P=n.Vw(this),this.b=n.Qw(this,t,P),A&&n.xi(A,this.b),e=0;e<P.a.length;e++)C.u3(t,this.b.c,this.b.d+1+r*5/6+r*se*e,(w.HAb(e,P.a.length),P.a[e]));},L.Jb=function(t){return n.Uw(this,t);},L.Kb=function(){var t;return t=new h.Mkb(),h.Kkb(t,' text="'+n.Tw(this.e)+'"'),h.Kkb(t,' x="'+this.i[0].a+'"'),h.Kkb(t,' y="'+this.i[0].b+'"'),this.c!=9&&h.Kkb(t,' size="'+p.mlb(new p.plb(this.c,new p.Wmb()))+'"'),this.d!=0&&h.Kkb(t,' style="'+this.d+'"'),t.a;},L.Lb=function(){return "text";},L.Mb=function(t){var A,e;A=(e=n.Vw(this),n.Qw(this,t,e)),C.H3(t,C.D3(t)),C.x3(t,A.c,A.d,A.b,A.a);},L.Nb=function(t){n.me(this,t),this.c*=t;},L.Pb=function(t,A){n.qe(this,t,A),this.c=this.f*t;},L.Qb=function(t,A){n.re(this,t,A),this.f=this.c;},L.a=false,L.c=0,L.d=0,L.f=0,Q.V8=N(94),n.px=function(t){t.b=new s.Mj(),t.a=new s.Mj();},n.qx=function(t,A){var e,r;return e=n.Xv(t,A),e==-1?-1:(r=t.b.a.length,s.sj(t.b,A),s.rj(t.a,e,h.sjb(r)),r);},n.rx=function(){n.Zv.call(this),n.px(this);},I(282,102,ii,n.rx),L.bc=function(t){var A;return A=n.Yv(this,t),A==-1?-1:s.xj(this.a,A).a;},Q.W8=N(282),q.tx=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st;for(Dt=n.Ep(t.e),Qt=s.xj(Dt.k,A).length,lt=s.xj(Dt.j,A),ut=s.xj(Dt.k,A),V=true,F=1;F<Qt;F++)if(q.Ix(t.b,ut[F])!=q.Ix(t.b,ut[0])){V=false;break;}if(V){for(i=(Qt-2)*xn/Qt,S=0;S<Qt;S++)q.yx(t,lt[S],ut[S],ut[S==0?Qt-1:S-1],i);return;}for(K=B(Q.R7,w0,6,Qt,15,1),P=0,R=0;R<Qt;R++)K[R]=n.Qo(t.e,lt[R])==0?1.9106119386631804:n.Qo(t.e,lt[R])==1?2.094499897810209:xn,P+=K[R];for(r=((Qt-2)*xn-P)/Qt,E=0;E<Qt;E++)K[E]+=r;for(U=B(Q.R7,w0,6,Qt,15,1),O=1;O<Qt;O++)U[O]=U[O-1]+xn-K[O];for(u=B(Q.R7,w0,6,Qt,15,1),D=0;D<100;D++){for(bt=0,st=0,S=0;S<Qt;S++)bt+=q.Ix(t.b,ut[S])*d.Math.sin(U[S]),st+=q.Ix(t.b,ut[S])*d.Math.cos(U[S]);for(c=n.no(0,0,bt,st),g=d.Math.sqrt(bt*bt+st*st),ot=-1,At=0,x=0;x<Qt;x++)G=x==0?Qt-1:x-1,J=x+1==Qt?0:x+1,o=n.oo(U[x],U[G]),l=n.oo(U[J],U[x]),it=n.oo(K[x],K[J]),u[x]=2*o-2*l+2*it,b=d.Math.cos(U[x]-wP-c),T=g*b-Di*u[x],d.Math.abs(T)>d.Math.abs(At)&&(At=T,ot=x);y=d.Math.exp(-5*D/100),U[ot]+=y*At;}for(k=0;k<Qt;k++)G=k==0?Qt-1:k-1,e=U[G]+xn-U[k],e>N1&&(e-=N1),q.yx(t,lt[k],ut[G],ut[k],e);},q.ux=function(t,A){var e,r,P,i,D,u,o,l,U;for(o=n.Ep(t.e),l=s.xj(o.k,A).length,u=s.xj(o.k,A),U=false,P=0;P<l;P++)if(t.f[u[P]]==0){U=true;break;}if(U){for(e=(180*l-360)/l*.01745329238474369,D=s.xj(o.j,A),i=0;i<l;i++)q.yx(t,D[i],u[i==0?l-1:i-1],u[i],e);if(l<=4)for(r=0;r<l;r++)t.f[u[r]]+=l;}},q.vx=function(t,A,e,r){var P,i,D,u;for(i=-1,D=-1,u=0;u<n.Lo(t.e,A);u++)if(P=n.Zo(t.e,A,u),P==e||P==r)if(D==-1)D=u;else {i=u;break;}return t.a[A][i][D];},q.wx=function(t,A,e,r){return e<r?t.a[A][r][e]:t.a[A][e][r];},q.xx=function(t,A){var e,r,P,i,D,u,o;for(r=B(Q.xfb,Tt,6,n.$o(t.e,A),16,1),o=0,P=0;P<n.$o(t.e,A);P++){for(u=0,D=-1,i=0;i<n.$o(t.e,A);i++)r[i]||(e=n._o(t.e,A,i),u<t.f[e]&&(u=t.f[e],D=i));if(u==0)return o;o<<=8,o+=u,r[D]=true;}return o;},q.yx=function(t,A,e,r,P){var i,D,u,o;for(i=-1,D=-1,o=0;o<n.Lo(t.e,A);o++)if(u=n._o(t.e,A,o),u==e||u==r)if(i==-1)i=o;else {D=o;break;}t.a[A][D][i]==0&&(t.a[A][D][i]=P,t.d[A]+=P,++t.c[A]);},q.zx=function(t,A,e,r,P){var i,D,u,o,l,U;for(i=n.Lo(t.e,A),o=n.$o(t.e,A),l=1;l<i;l++)for(D=l<o?t.f[n._o(t.e,A,l)]:0,U=0;U<l;U++)t.a[A][l][U]==0&&(u=U<o?t.f[n._o(t.e,A,U)]:0,(D==e&&u==r||D==r&&u==e)&&(t.a[A][l][U]=P));},q.Ax=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G;for(this.b=A,this.e=t,this.a=B(Q.R7,b0,467,this.e.f,0,3),P=0;P<this.e.f;P++)for(this.a[P]=B(Q.R7,{467:1,4:1,9:1,5:1,8:1},46,n.Lo(this.e,P),0,2),o=1;o<n.Lo(this.e,P);o++)this.a[P][o]=B(Q.R7,w0,6,o,15,1);for(this.c=B(Q.S7,H,6,this.e.f,15,1),this.d=B(Q.R7,w0,6,this.e.f,15,1),this.f=B(Q.S7,H,6,this.e.g,15,1),x=n.Ep(this.e),T=B(Q.xfb,Tt,6,x.j.a.length,16,1),n.at(x,T,B(Q.xfb,Tt,6,x.j.a.length,16,1),B(Q.S7,H,6,x.j.a.length,15,1),true),u=3;u<=7;u++)for(F=0;F<x.j.a.length;F++)G=s.xj(x.k,F).length,G==u&&(T[F]?q.tx(this,F):G<=4&&q.ux(this,F));for(y=B(Q.xfb,Tt,6,this.e.f,16,1),k=0;k<x.j.a.length;k++)if(T[k])for(R=s.xj(x.j,k),E=0,O=R.length;E<O;++E)S=R[E],y[S]=true;for(i=j(v(Q.S7,1),H,6,15,[0,0,1,3,6,10,15,21]),r=0;r<this.e.f;r++){if(D=n.Lo(this.e,r),D>4){for(o=1;o<D;o++)for(b=0;b<o;b++)this.a[r][o][b]=wP;this.c[r]=i[D];continue;}if(this.c[r]!=i[D])if(n.iq(this.e,r)&&(y[r]||n.Uo(this.e,r)<=4&&n.Qo(this.e,r)>0)){if(D>2){if(this.c[r]==1)n.Tl(this.e,r)<=14?e=(ui-this.d[r])/2:e=oi+(oi-this.d[r])*.18000000715255737;else if(e=ui-this.d[r],D>3){if(this.c[r]==2){for(c=B(Q.xfb,Tt,6,n.Lo(this.e,r),16,1),l=1;l<D;l++)for(g=0;g<l;g++)this.a[r][l][g]!=0&&(c[l]=!c[l],c[g]=!c[g]);for(U=0;U<D;U++)if(c[U]){for(g=U+1;g<D;g++)if(c[g]){this.a[r][g][U]=e;break;}break;}}e=wP;}for(o=1;o<D;o++)for(b=0;b<o;b++)this.a[r][o][b]==0&&(this.a[r][o][b]=e);}}else if(n.iq(this.e,r)&&n.Uo(this.e,r)<=4)switch(q.xx(this,r)){case 771:q.zx(this,r,0,3,2.0653998851776123),q.zx(this,r,0,0,1.9814722631346626);break;case 1028:q.zx(this,r,0,4,1.9797999858856201),q.zx(this,r,0,0,1.94691481878138);break;case 393987:q.zx(this,r,0,3,Qi),q.zx(this,r,0,6,Qi),q.zx(this,r,3,3,1.7229016938441077);break;case 459779:q.zx(this,r,0,3,gP),q.zx(this,r,0,4,gP),q.zx(this,r,0,7,gP),q.zx(this,r,3,4,1.9322539839360076);break;case 525316:q.zx(this,r,0,4,li),q.zx(this,r,0,8,li),q.zx(this,r,4,4,1.99944913298566);case 394758:q.zx(this,r,0,6,2.526099920272827);break;case 460550:q.zx(this,r,0,6,Ui),q.zx(this,r,0,7,Ui);break;case 526087:q.zx(this,r,0,7,Li),q.zx(this,r,0,8,Li);break;case 526344:q.zx(this,r,0,8,2.186300039291382);break;case 50529027:q.zx(this,r,3,3,2.4189000129699707);break;case 67371779:q.zx(this,r,3,4,2.2298998832702637);break;case 67372036:q.zx(this,r,4,4,2.094399929046631);break;case 101057283:q.zx(this,r,3,6,1.839926051241747),q.zx(this,r,3,3,2.9061476191098734);break;case 117834755:q.zx(this,r,3,4,2.812249087174905),q.zx(this,r,3,7,1.7910569124592968),q.zx(this,r,4,6,2.1224948975613245);break;case 134677507:q.zx(this,r,3,4,2.642428360523752),q.zx(this,r,3,8,2.027723514585844),q.zx(this,r,4,7,2.251474717631936);break;case 117900035:q.zx(this,r,3,7,2.109753935530918),q.zx(this,r,3,3,3.1052897491356646);break;case 117900292:q.zx(this,r,4,7,2.090729910747413),q.zx(this,r,4,4,2.551671293386306);break;case 134743044:q.zx(this,r,4,8,2.139250042271712),q.zx(this,r,4,4,2.3520055858942612);}else for(e=n.Tl(this.e,r)>10?si:n.Qo(this.e,r)==2?xn:n.$p(this.e,r,true)?yi:n.Qo(this.e,r)==0?si:yi,o=1;o<D;o++)for(b=0;b<o;b++)this.a[r][o][b]=e;}},I(382,1,{},q.Ax),Q.X8=N(382),q.Hx=function(){q.Hx=i1,q.Fx=j(v(Q.xfb,1),Tt,6,16,[false,false,false,false,false,true,true,true,true,false,false,false,false,false,false,true,true]);},q.Ix=function(t,A){return t.a[A];},q.Jx=function(t){q.Hx();var A,e;for(n.bw(t,7),this.a=B(Q.R7,w0,6,t.r,15,1),this.b=B(Q.R7,w0,6,t.r,15,1),A=0;A<t.r;A++)e=q.Nx(t,A),e==-1?(this.a[A]=q.Ox(t,A),this.b[A]=q.Px(t,A)):(this.a[A]=e==-1?2.000499963760376:q.Dx[e],this.b[A]=e==-1?1:q.Ex[e]);},q.Kx=function(t,A){return A>=t.f?0:A<t.f&&n.ot(t.p,A)&&t.C[A]==6&&t.s[A]!=0?1:t.o[A];},q.Lx=function(t,A,e,r,P,i,D,u,o){var l,U,y,T,c;return T=r<q.Fx.length&&q.Fx[r]?d.Math.min(3,i):0,c=P<q.Fx.length&&q.Fx[P]?d.Math.min(3,D):0,u=d.Math.min(7,u),o=d.Math.min(7,o),l=u+(T<<3)+(d.Math.min(127,r)<<5),U=o+(c<<3)+(d.Math.min(127,P)<<5),y=e?8:A?8+t:t,(y<<24)+(l<U?(l<<12)+U:(U<<12)+l);},q.Mx=function(t,A,e,r,P,i,D,u,o,l){var U;return U=q.Rx(q.Lx(t,A,e,r,P,i,D,u,o)),!l&&U==-1&&(U=q.Rx(q.Lx(t,A,e,r,P,i,D,7,7)),U==-1&&(U=q.Rx(q.Lx(t,A,e,r,P,3,3,7,7)))),U;},q.Nx=function(t,A){var e,r,P,i,D,u;return e=t.D[0][A],r=t.D[1][A],P=t.C[e],i=t.C[r],D=t.j[e],u=t.j[r],q.Mx(n.fm(t,A),A<t.g&&n.pt(t.p,A),A<t.g&&(n.ut(t.p,A)||t.H[A]==8),P,i,q.Kx(t,e),q.Kx(t,r),D,u,false);},q.Ox=function(t,A){var e,r;return e=n.Tl(t,t.D[0][A]),r=n.Tl(t,t.D[1][A]),(e<(q.qz(),q.oz).length?q.oz[e]:e<q.pz.length?ye*q.pz[e]:or)+(r<q.oz.length?q.oz[r]:r<q.pz.length?ye*q.pz[r]:or);},q.Px=function(t,A){var e,r;return e=n.Tl(t,t.D[0][A]),r=n.Tl(t,t.D[1][A]),(e<(q.qz(),q.oz).length?BP:.125)+(r<q.oz.length?BP:.125);},q.Qx=function(){var t,A,e,r,P,i,D;if(!q.Gx)try{for(t=q.Ey("bondLengthData.txt"),e=Ct.fhb(t),A=e==null?0:h.Oib(e,e1,Ot),q.Cx=B(Q.S7,H,6,A,15,1),q.Dx=B(Q.R7,w0,6,A,15,1),q.Ex=B(Q.R7,w0,6,A,15,1),q.Bx=B(Q.S7,H,6,A,15,1),P=0;P<A;P++)if(D=Ct.fhb(t),D!=null&&(i=h.qkb(D,"\\t"),i.length==4))try{q.Cx[P]=h.Oib(i[0],e1,Ot),q.Dx[P]=I0(i[1]),q.Ex[P]=I0(i[2]),q.Bx[P]=h.Oib(i[3],e1,Ot);}catch(u){if(u=M1(u),Zt(u,40))break;throw Lt(u);}q.Gx=!0;}catch(u){if(u=M1(u),Zt(u,75))r=u,h.MB(r,(h.Rkb(),h.Qkb),"");else throw Lt(u);}},q.Rx=function(t){var A,e,r,P;for(q.Gx||q.Qx(),P=Z1,r=B1,e=0;e<13;e++){if(A=P>=q.Cx.length||t<q.Cx[P]?-1:t==q.Cx[P]?0:1,A==0)return P;P=A<0?P-r:P+r,r=r/2|0;}return  -1;},I(381,1,{},q.Jx),q.Gx=false,Q.Y8=N(381),q.Sx=function(t,A){t.q=q.Ry(new q.Uy(t.o,A),t);},q.Tx=function(t,A){var e,r,P,i,D,u,o,l,U;return e=q.Zx(t,A[0]),r=q.Zx(t,A[1]),P=q.Zx(t,A[2]),i=q.Zx(t,A[3]),o=new n.oi(r.a-e.a,r.b-e.b,r.c-e.c),l=new n.oi(P.a-r.a,P.b-r.b,P.c-r.c),U=new n.oi(i.a-P.a,i.b-P.b,i.c-P.c),D=new n.oi(o.b*l.c-o.c*l.b,-(o.a*l.c-o.c*l.a),o.a*l.b-o.b*l.a),u=new n.oi(l.b*U.c-l.c*U.b,-(l.a*U.c-l.c*U.a),l.a*U.b-l.b*U.a),-d.Math.atan2(d.Math.sqrt(l.a*l.a+l.b*l.b+l.c*l.c)*(o.a*u.a+o.b*u.b+o.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},q.Ux=function(t,A){return q.Xx(t,A),q.Fy(t.q,A.q);},q.Vx=function(t,A){var e;for(e=0;e<A.q;e++)n.ji(t.k[e],A.J[e]);},q.Wx=function(t,A){var e;for(e=0;e<A.k.length;e++)n.ji(t.k[e],A.k[e]);A.j==null?t.j=null:t.j=s.Crb(A.j,A.j.length);},q.Xx=function(t,A){var e;(!t.q||!A.q)&&(e=new q.Ty(t.o),!t.q&&(t.q=q.Ry(e,t)),!A.q&&(A.q=q.Ry(e,A)));},q.Yx=function(t,A){return t.j==null?-1:t.j[A];},q.Zx=function(t,A){return t.k[A];},q.$x=function(t){return t.p==null?t.o.P:t.p;},q._x=function(t,A){return t.k[A].a;},q.ay=function(t,A){return t.k[A].b;},q.by=function(t,A){return t.k[A].c;},q.cy=function(t,A,e){for(t.j==null&&(t.j=B(Q.wfb,Yt,6,t.o.r,15,1),s.Irb(t.j));e<0;)e=e+360<<16>>16;for(;e>=360;)e=e-360<<16>>16;t.j[A]=e;},q.dy=function(t,A,e){n.ji(t.k[A],e);},q.ey=function(t,A,e){t.k[A]=e;},q.fy=function(t,A){t.p=A;},q.gy=function(t,A,e){t.k[A].a=e;},q.hy=function(t,A,e){t.k[A].b=e;},q.iy=function(t,A,e){t.k[A].c=e;},q.jy=function(t,A){var e;for(!A&&(A=n.fw(t.o)),A.T&=-249,e=0;e<A.q;e++)n.ji(A.J[e],t.k[e]);return t.p!=null&&n.Un(A,t.p),A;},q.ky=function(t){var A;for(this.o=t,this.k=B(Q.j8,En,29,t.q,0,1),A=0;A<t.q;A++)this.k[A]=new n.pi(t.J[A]);this.n=NaN;},q.ly=function(t){q.my.call(this,t,t.o);},q.my=function(t,A){var e;for(this.o=A,this.k=B(Q.j8,En,29,t.k.length,0,1),e=0;e<this.k.length;e++)this.k[e]=new n.pi(t.k[e]);t.j!=null&&(this.j=s.Crb(t.j,t.j.length)),this.p=t.p==null||h.bkb(t.p)?t.p:h.$jb(t.p,Rn),this.n=NaN;},I(63,1,{63:1,32:1},q.ly),L.Sb=function(t){return q.Ux(this,t);},L.n=0,Q.Z8=N(63),q.qy=function(t,A){var e,r,P,i,D;return A==null?null:(D=s.gqb(t.c,A),D||(r=w.UAb("<").length,(h.ckb(w.UAb(A).substr(w.UAb(A).length-r,r),"<")||(P=w.UAb("-").length,h.ckb(w.UAb(A).substr(w.UAb(A).length-P,P),"-")))&&(D=s.gqb(t.c,(i=w.UAb("<").length,h.ckb(w.UAb(A).substr(w.UAb(A).length-i,i),"<")?h.tkb(A,0,w.UAb(A).length-1)+">":(e=w.UAb("-").length,h.ckb(w.UAb(A).substr(w.UAb(A).length-e,e),"-")?h.tkb(A,0,w.UAb(A).length-1)+"+":A))),D)?(D=new q.mz(D),s.Rwb(t.c,A,D),D):null));},q.ry=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E;t.b|=A;try{for(R=q.Ey("torsionID.txt"),r=(A&1)==0?null:q.Ey("torsionAngle.txt"),T=(A&1)==0?null:q.Ey("torsionRange.txt"),u=(A&1)==0?null:q.Ey("torsionFrequency.txt"),i=(A&2)==0?null:q.Ey("torsionBins.txt"),E=Ct.fhb(R);E!=null;){if(S=s.gqb(t.c,E),S||(S=new q.lz((b=w.UAb("<").length,h.ckb(w.UAb(E).substr(w.UAb(E).length-b,b),"<")||(g=w.UAb(">").length,h.ckb(w.UAb(E).substr(w.UAb(E).length-g,g),">"))?0:(k=w.UAb("-").length,h.ckb(w.UAb(E).substr(w.UAb(E).length-k,k),"-")||(F=w.UAb("+").length,h.ckb(w.UAb(E).substr(w.UAb(E).length-F,F),"+"))?1:(c=w.UAb("=").length,h.ckb(w.UAb(E).substr(w.UAb(E).length-c,c),"=")?3:2)))),s.Rwb(t.c,E,S)),r)for(e=h.qkb(Ct.fhb(r),","),S.a=B(Q.wfb,Yt,6,e.length,15,1),l=0;l<e.length;l++)S.a[l]=h.Oib(e[l],Je,32767)<<16>>16;if(T)for(y=h.qkb(Ct.fhb(T),","),S.d=T1(Q.wfb,[P0,Yt],[16,6],15,[y.length,2],2),l=0;l<y.length;l++)U=h.ikb(y[l],45,1),S.d[l][0]=h.Oib(h.tkb(y[l],0,U),Je,32767)<<16>>16,S.d[l][1]=h.Oib(h.skb(y[l],U+1),Je,32767)<<16>>16;if(u)for(o=h.qkb(Ct.fhb(u),","),S.c=B(Q.wfb,Yt,6,o.length,15,1),l=0;l<o.length;l++)S.c[l]=h.Oib(o[l],-128,127)<<24>>24;if(i)for(P=h.qkb(Ct.fhb(i),","),S.b=B(Q.O7,Y,6,P.length,15,1),l=0;l<P.length;l++)S.b[l]=h.Oib(P[l],-128,127)<<24>>24;E=Ct.fhb(R);}}catch(O){if(O=M1(O),Zt(O,75))D=O,h.MB(D,(h.Rkb(),h.Qkb),"");else throw Lt(O);}},q.sy=function(t){var A,e,r,P;for(t.a=15,P=(e=new s.Gpb(t.c).a.cd().Tb(),new s.Npb(e));P.a.ed();)r=(A=P.a.fd(),A.jd()),q.jz(r);},q.ty=function(){this.c=new s.Zwb(),this.b=0;},q.uy=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k;for(r=B(Q.Q7,jt,6,2,15,1),c=q.xy(t,e),o=0;o<2;o++)if(e[3*o]!=-1){for(i=2-o,b=3-3*o,l=0,T=0;T<3;T++)D=n.Zo(t,e[i],T),D!=c[1-o]&&t.C[D]!=1&&(e[b]=D,A?r[l++]=q.Tx(A,e):r[l++]=n.el(t,e));return e[b]=-1,q.wy(r);}for(P=B(Q.Q7,jt,6,2,15,1),U=0,u=0;u<3;u++)if(g=n.Zo(t,e[1],u),g!=c[0]&&t.C[g]!=1){for(e[0]=g,y=0,T=0;T<3;T++)k=n.Zo(t,e[2],T),k!=c[1]&&t.C[k]!=1&&(e[3]=k,A?P[y++]=q.Tx(A,e):P[y++]=n.el(t,e));r[U++]=q.wy(P);}return e[0]=-1,e[3]=-1,q.wy(r);},q.vy=function(t,A){return A[0]!=-1&&A[3]!=-1?q.Tx(t,A):q.uy(t.o,t,A);},q.wy=function(t){var A,e;return e=(t[1]+t[0])/2,A=d.Math.abs(t[1]-t[0]),A>t1?e:e<0?e+t1:e-t1;},q.xy=function(t,A){var e,r,P;return P=B(Q.S7,H,6,2,15,1),n.Xo(t,A[1],A[2])!=-1?(P[0]=A[2],P[1]=A[1]):(e=B(Q.S7,H,6,16,15,1),r=n.yp(t,e,A[1],A[2],15,null,null),P[0]=e[1],P[1]=e[r-1]),P;},q.yy=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F;for(n.bw(t,7),l=0,P=0;P<t.g;P++)n.fm(t,P)==1&&!(P<t.g&&n.pt(t.p,P))&&n.tp(t,t.D[0][P])>1&&n.tp(t,t.D[1][P])>1&&(t.F[P]&64)==0&&!((t.F[P]&128)!=0&&(t.p&&P<t.g?n.ft(t.p,P):0)<=5)&&(A[P]=true,++l);for(D=B(Q.S7,H,6,2,15,1),c=B(Q.S7,H,6,2,15,1),i=null,r=0;r<t.g;r++)if(A[r]&&(i==null||!i[r])){for(e=0,y=0;y<2;y++)for(D[y]=t.D[y][r],c[y]=t.D[1-y][r],g=c[y];n.Qo(t,D[y])==2&&n.tp(t,D[y])==2&&n.Tl(t,D[y])<10;){for(T=0;T<2;T++)if(u=n.Zo(t,D[y],T),u!=c[y]){o=n._o(t,D[y],T),A[o]&&n.fm(t,o)==1&&(A[o]=false,--l),c[y]=D[y],D[y]=u,++e;break;}if(D[y]==g){e=0;break;}}e!=0&&(A[r]=false,--l,n.tp(t,D[0])>1&&n.tp(t,D[1])>1&&(k=n.Lp(t,c[0],D[0]),F=n.Lp(t,c[1],D[1]),U=k<F?0:1,b=n.Xo(t,c[U],D[U]),i==null&&(i=B(Q.xfb,Tt,6,t.g,16,1)),i[b]=true,A[b]=true,++l));}return l;},q.zy=function(t){var A;return A=q.qy(q.py,t),A?q.gz(A):null;},q.Ay=function(t,A,e,r){return q.az(r,t,A),e!=null&&(e[0]=r.f[0],e[1]=r.b[0],e[2]=r.b[1],e[3]=r.f[1]),r.d;},q.By=function(t){var A;return A=q.qy(q.py,t),A?q.hz(A):null;},q.Cy=function(t){var A;return A=q.qy(q.py,t),A?q.fz(A):null;},q.Dy=function(){var t;!q.py&&(q.py=new q.ty()),(q.py.b&1)!=0&&q.py.a!=15&&h.Rkb(),t=1&~q.py.b,t!=0&&(q.ry(q.py,t),t==1&&q.sy(q.py));},q.Ey=function(t){var A,e,r;if(q.oy==null){if(A=_t.eBb("/resources/csd/"+t),A)return q.oy="csd/",new Ct.ghb(new Ct.lhb((r=A,O1.Enb(),r)));q.oy="cod/";}return new Ct.ghb(new Ct.lhb((e=_t.eBb("/resources/"+q.oy+t),O1.Enb(),e)));},I(372,1,{},q.ty),L.a=0,L.b=0,Q.$8=N(372),q.Fy=function(t,A){var e,r;for(r=0;r<t.b.length;r++)if(e=d.Math.abs(t.b[r]-A.b[r]),e>Qr&&e<t.a[r]-Qr)return e<t.a[r]/2^t.b[r]<A.b[r]?1:-1;return 0;},q.Gy=function(t,A){var e,r;for(r=0;r<t.b.length;r++)if(e=d.Math.abs(t.b[r]-A.b[r]),e>Qr&&e<t.a[r]-Qr)return  false;return  true;},q.Hy=function(t,A){this.b=t,this.a=A;},I(256,1,{256:1,32:1},q.Hy),L.Sb=function(t){return q.Fy(this,t);},L.pb=function(){var t,A;for(A=new h.Mkb(),t=0;t<this.b.length;t++)A.a+=t==0?"Torsions: ":", ",h.Kkb(A,(Ut.D4(),Ut.E4(this.b[t],3)+"("+pt(d.Math.round(this.a[t]+.5))+")"));return A.a;},Q.a9=N(256),q.My=function(){q.My=i1,q.Ly=j(v(Q.Q7,1),jt,6,15,[N1,t1,qn,c0]),q.Ky=j(v(Q.S7,2),E1,7,0,[j(v(Q.S7,1),H,6,15,[0,0,0,2]),j(v(Q.S7,1),H,6,15,[0,0,1,2]),j(v(Q.S7,1),H,6,15,[0,1,1,3]),j(v(Q.S7,1),H,6,15,[2,2,3,2])]);},q.Ny=function(t,A,e,r){var P,i,D,u;if(r==1&&!q.Sy(t,A)){if(n.$o(t.b,A)==3)return  -1;if(n.$o(t.b,A)==3)return P=q.Qy(t,A,e),n.mw(t.b,P[0])==n.mw(t.b,P[1])?P[2]:n.mw(t.b,P[0])==n.mw(t.b,P[2])?P[1]:P[0];}for(D=-1,u=-1,i=0;i<n.$o(t.b,A);i++)P=n.Zo(t.b,A,i),P!=e&&D<n.mw(t.b,P)&&(D=n.mw(t.b,P),u=P);return u;},q.Oy=function(t){var A,e,r,P,i;for(t.a=T1(Q.S7,[E1,H],[7,6],15,[t.d.length,4],2),t.c=T1(Q.S7,[E1,H],[7,6],15,[t.d.length,2],2),t.e=B(Q.S7,H,6,t.d.length,15,1),A=B(Q.S7,H,6,2,15,1),P=0;P<t.d.length;P++){for(i=0;i<2;i++)A[0]=n.Zl(t.b,1-i,t.d[P]),A[1]=n.Zl(t.b,i,t.d[P]),q.Yy(t.b,A[1])&&q.Wy(t.b,A[0],A,null),t.a[P][1+i]=A[1],t.c[P][i]=A[0];e=q.Py(t,t.a[P][1],t.c[P][0]),r=q.Py(t,t.a[P][2],t.c[P][1]),t.e[P]=q.Ky[e][r],t.a[P][0]=q.Ny(t,t.a[P][1],t.c[P][0],e),t.a[P][3]=q.Ny(t,t.a[P][2],t.c[P][1],r);}},q.Py=function(t,A,e){var r;if(n.$o(t.b,A)==2)return 1;if(r=q.Qy(t,A,e),n.$o(t.b,A)==3)return n.mw(t.b,r[0])==n.mw(t.b,r[1])?q.Sy(t,A)?2:1:q.Sy(t,A)?1:0;if(n.$o(t.b,A)==4){if(n.mw(t.b,r[0])==n.mw(t.b,r[1])&&n.mw(t.b,r[0])==n.mw(t.b,r[2]))return 3;if(n.mw(t.b,r[0])==n.mw(t.b,r[1])||n.mw(t.b,r[0])==n.mw(t.b,r[2])||n.mw(t.b,r[1])==n.mw(t.b,r[2]))return 1;}return 0;},q.Qy=function(t,A,e){var r,P,i;for(i=0,r=B(Q.S7,H,6,n.$o(t.b,A)-1,15,1),P=0;P<n.$o(t.b,A);P++)n.Zo(t.b,A,P)!=e&&(r[i++]=n.Zo(t.b,A,P));return r;},q.Ry=function(t,A){var e,r,P;for(P=B(Q.R7,w0,6,t.d.length,15,1),r=B(Q.R7,w0,6,t.d.length,15,1),e=0;e<t.d.length;e++)P[e]=q.Xy(q.vy(A,t.a[e]),t.e[e]),r[e]=q.Ly[t.e[e]];return new q.Hy(P,r);},q.Sy=function(t,A){return !!(n.Qo(t.b,A)==1&&n.Tl(t.b,A)<10||n.Tp(t.b,A)||n.$p(t.b,A,true));},q.Ty=function(t){q.My(),this.b=t,n.bw(this.b,63),this.d=q.Vy(t),q.Oy(this);},q.Uy=function(t,A){q.My(),this.b=t,n.bw(this.b,63),this.d=A,q.Oy(this);},q.Vy=function(t){q.My();var A,e,r,P,i;for(n.bw(t,7),P=B(Q.xfb,Tt,6,t.g,16,1),r=0,e=0;e<t.g;e++)q.Zy(t,e)&&(P[e]=true,++r);for(i=B(Q.S7,H,6,r,15,1),r=0,A=0;A<t.g;A++)P[A]&&(i[r++]=A);return i;},q.Wy=function(t,A,e,r){var P,i,D;for(P=0;P<2;P++)if(i=n.Zo(t,e[1],P),i!=e[0])return i==A||(D=n._o(t,e[1],P),e[0]=e[1],e[1]=i,t.j[i]==1)?false:(r!=null&&!n.Km(t,e[0])&&!n.Km(t,e[1])&&(r[0]=d.Math.max(r[0],D)),t.j[i]==2&&t.o[i]==2&&t.C[i]<=7?q.Wy(t,A,e,r):true);return  false;},q.Xy=function(t,A){var e;for(e=q.Ly[A]/2;t<-e;)t+=N1;for(;t>=e;)t-=q.Ly[A];return t;},q.Yy=function(t,A){return t.j[A]==2&&t.o[A]==2&&t.C[A]<=7;},q.Zy=function(t,A){var e,r,P,i,D,u,o,l,U,y,T;if(n.fm(t,A)!=1||A<t.g&&n.pt(t.p,A)||(t.p&&A<t.g?n.ft(t.p,A):0)==3)return  false;for(e=B(Q.S7,H,6,2,15,1),u=0;u<2;u++)if(e[u]=t.D[u][A],n.Km(t,e[u])||n.tp(t,e[u])<=1)return  false;if(q.Yy(t,e[0])||q.Yy(t,e[1])){for(y=B(Q.S7,H,6,1,15,1),y[0]=A,r=B(Q.S7,E1,7,2,0,2),o=0;o<2;o++)if(q.Yy(t,e[o])&&(r[o]=B(Q.S7,H,6,2,15,1),r[o][0]=e[1-o],r[o][1]=e[o],!q.Wy(t,e[1-o],r[o],y)))return  false;if(A!=y[0])return  false;for(l=0;l<2;l++)r[l]!=null&&(e[l]=r[l][1]);}for(D=0;D<2;D++){if(i=n.$o(t,e[D]),i==1)return  false;for(T=0,U=0;U<i;U++)P=n.Zo(t,e[D],U),(t.u[P]&r0)!=0||++T;if(T<2)return  false;}return  true;},I(254,1,{},q.Ty,q.Uy),Q._8=N(254),q._y=function(){q._y=i1,q.$y=j(v(Q.S7,2),E1,7,0,[j(v(Q.S7,1),H,6,15,[0,0,1]),j(v(Q.S7,1),H,6,15,[0,2,3]),j(v(Q.S7,1),H,6,15,[1,3,3])]);},q.az=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt,st,wt,ct,Vt,Et,Ht,Jt,Nt,Gt;if(n._v(t.c),t.d=null,n.bw(A,63),n.fm(A,e)!=1||e<A.g&&n.pt(A.p,e)||n.Tl(A,A.D[0][e])==1||n.Tl(A,A.D[1][e])==1||(lt=(A.F[e]&128)!=0,lt&&(A.p&&e<A.g?n.ft(A.p,e):0)<6))return  false;for(P=B(Q.xfb,Tt,6,A.f,16,1),t.a=0,G=0;G<2;G++){for(t.b[G]=A.D[G][e],t.e[G]=A.D[1-G][e];n.Qo(A,t.b[G])==2&&n.tp(A,t.b[G])==2&&n.Tl(A,t.b[G])<10;)for(ut=0;ut<n.$o(A,t.b[G]);ut++)if(D=n.Zo(A,t.b[G],ut),D!=t.e[G]&&A.C[D]!=1){if(A.j[D]==1||t.a==16)return  false;P[t.b[G]]=true,t.e[G]=t.b[G],t.b[G]=D,++t.a;break;}if(Dt=n.tp(A,t.b[G]),Dt>4||Dt==1)return  false;P[t.b[G]]=true;}for(J=0;J<2;J++)for(ut=0;ut<n.$o(A,t.b[J]);ut++)D=n.Zo(A,t.b[J],ut),A.C[D]!=1&&(P[D]=true);for(t.g=B(Q.S7,H,6,A.f,15,1),n.Co(A,t.c,P,true,t.g),V=0;V<t.g.length;V++)t.g[V]!=-1&&(t.i[t.g[V]]=V);if(n.On(t.c,true),lt&&(i=n.Xo(t.c,t.g[t.b[0]],t.g[t.b[1]]),i!=-1)){for(n.Ln(t.c,i,256,true),Et=(n.bw(A,7),A.p),Vt=0;Vt<Et.j.a.length;Vt++)if(n.rt(Et,Vt,e)){for(At=0;At<2;At++)for(ut=0;ut<n.$o(A,t.b[At]);ut++)if(D=n.Zo(A,t.b[At],ut),D!=t.e[At]&&n.qt(Et,Vt,D)&&A.C[D]!=1){n.Ln(t.c,n.Xo(t.c,t.g[t.b[At]],t.g[D]),256,true);break;}}}for(ot=0;ot<2;ot++){for(n.$p(A,t.b[ot],true)&&n.vn(t.c,t.g[t.b[ot]],NA,true),o=false,ut=0;ut<n.$o(A,t.b[ot]);ut++)D=n.Zo(A,t.b[ot],ut),D!=t.e[ot]&&A.C[D]!=1&&(F=n.Xo(t.c,t.g[t.b[ot]],t.g[D]),n.im(t.c,F)==8?o=true:A.C[D]==6&&!n.Tp(A,t.b[ot])&&(k=D<A.f&&n.ot(A.p,D)?2:4,n.vn(t.c,t.g[D],k,true)),u=n._o(A,t.b[ot],ut),Ht=A.p&&u<A.g?n.ft(A.p,u):0,(Ht==3||Ht==4)&&n.Ln(t.c,F,Ht<<17,true),(u<A.g&&n.pt(A.p,u)||n.ap(A,t.b[ot],ut)==2)&&(Dt=n.tp(A,D),O=Dt==3,!O&&Dt==2&&(A.u[D]&x0)==0&&(O=n.Mp(A,t.b[1-ot],u)!=-1),O?n.vn(t.c,t.g[D],Cn,true):u<A.g&&n.pt(A.p,u)&&n.vn(t.c,t.g[D],pn,true)),n.ap(A,t.b[ot],ut)==1&&(n.tp(A,D)==4?n.vn(t.c,t.g[D],ln,true):A.C[D]==6&&n.vn(t.c,t.g[D],j0,true)));o||(n.Tp(A,t.b[ot])?n.vn(t.c,t.g[t.b[ot]],2,true):n.vn(t.c,t.g[t.b[ot]],4,true));}for(n.bw(t.c,191),x=0;x<2;x++)if(S=t.g[t.b[x]],n.Nl(t.c,S)==3){if(n.Nl(A,t.b[x])==3)return  false;Qt=n.Ro(t.c,S),n.Mn(t.c,Qt,257),n.Zl(t.c,0,Qt)!=S&&(n.Dn(t.c,1,Qt,n.Zl(t.c,0,Qt)),n.Dn(t.c,0,Qt,S)),n.bw(t.c,191);}if(T=t.g[t.b[0]],c=t.g[t.b[1]],wt=t.g[t.e[0]],ct=t.g[t.e[1]],l=n.Hl(t.c,T),U=n.Hl(t.c,c),n.zm(t.c,T)&&n.zm(t.c,c)&&(l!=0||U!=0)&&(l!=U||n.Gl(t.c,T)!=n.Gl(t.c,c)))return  false;if(y=false,n.zm(t.c,T)&&l!=0&&(n.on(t.c,T,0,-1),y=true),n.zm(t.c,c)&&U!=0&&(n.on(t.c,c,0,-1),y=true),y&&n.bw(t.c,191),bt=n.mw(t.c,T),st=n.mw(t.c,c),bt<st&&(Gt=t.b[0],t.b[0]=t.b[1],t.b[1]=Gt,Gt=t.e[0],t.e[0]=t.e[1],t.e[1]=Gt,Gt=T,T=c,c=Gt,Gt=wt,wt=ct,ct=Gt),it=false,(n.zm(t.c,T)||n.zm(t.c,c))&&(n.zm(t.c,T)?it=n.cw(t.c,T)==1:n.zm(t.c,c)&&(it=n.cw(t.c,c)==1),it)){for(r=0;r<t.c.q;r++)n.zn(t.c,r,-n.Ql(t.c,r));n.bw(t.c,191);}return b=q.cz(t,T,wt),g=q.cz(t,c,ct),t.f[0]=b==-1?-1:t.i[b],t.f[1]=g==-1?-1:t.i[g],K=n.jw(t.c),K==null?false:(R=q.bz(t,T,wt),E=q.bz(t,c,ct),R==0&&E==0&&(t.c.I&fA)==r0?Nt=2:Nt=q.$y[R][E],Jt=Nt==0?it?"<":">":Nt==1?it?"-":"+":Nt==3?"=":"",t.d=K+Jt,true);},q.bz=function(t,A,e){var r,P,i,D;if(n.$o(t.c,A)==2)return 1;if(r=q.dz(t,A,e),n.$o(t.c,A)==3)return n.mw(t.c,r[0])==n.mw(t.c,r[1])?n.Qo(t.c,A)==1&&n.Tl(t.c,A)<10||n.Tp(t.c,A)||n.$p(t.c,A,true)?2:1:n.Qo(t.c,A)==1&&n.Tl(t.c,A)<10||n.Tp(t.c,A)||n.$p(t.c,A,true)?1:0;if(n.$o(t.c,A)==4){for(P=0;P<r.length;P++)for(D=n.mw(t.c,r[P]),i=P+1;i<r.length;i++)if(D==n.mw(t.c,r[i]))return 1;}return 0;},q.cz=function(t,A,e){var r,P,i,D,u,o,l,U,y,T;for(U=-1,y=-1,T=-1,i=B(Q.xfb,Tt,6,n.$o(t.c,A),16,1),o=0;o<n.$o(t.c,A);o++)if(!i[o]&&(P=n.Zo(t.c,A,o),P!=e&&(D=n.mw(t.c,P),y<D))){for(u=false,l=o+1;l<n.$o(t.c,A);l++)if(r=n.Zo(t.c,A,l),r!=e&&n.mw(t.c,r)==D){if(i[l]=true,u)return P;u=true;}u?T=P:(y=D,U=P);}return U==-1&&(n.Qo(t.c,A)==1&&n.Tl(t.c,A)<10||n.Tp(t.c,A)||n.$p(t.c,A,true))?T:U;},q.dz=function(t,A,e){var r,P,i;for(i=0,r=B(Q.S7,H,6,n.$o(t.c,A)-1,15,1),P=0;P<n.$o(t.c,A);P++)n.Zo(t.c,A,P)!=e&&(r[i++]=n.Zo(t.c,A,P));return r;},q.ez=function(){q._y(),this.c=new n.uw(24,29),this.i=B(Q.S7,H,6,24,15,1),this.b=B(Q.S7,H,6,2,15,1),this.e=B(Q.S7,H,6,2,15,1),this.f=B(Q.S7,H,6,2,15,1);},I(250,1,{},q.ez),L.a=0,Q.b9=N(250),q.fz=function(t){var A,e,r,P,i,D,u,o,l;switch(A=null,t.e){case 1:for(A=B(Q.wfb,Yt,6,2*t.a.length,15,1),r=0;r<t.a.length;r++)A[r]=t.a[r],A[t.a.length+r]=180+t.a[r]<<16>>16;return A;case 2:for(P=t.a[0]==0?1:0,i=t.a[t.a.length-1]==180?t.a.length-1:t.a.length,l=i-P,A=B(Q.wfb,Yt,6,t.a.length+l,15,1),D=0;D<t.a.length;D++)A[D]=t.a[D];for(u=P;u<i;u++)A[A.length-1-u+P]=360-t.a[u]<<16>>16;return A;case 3:for(P=t.a[0]==0?1:0,i=t.a[t.a.length-1]==90?t.a.length-1:t.a.length,l=i-P,A=B(Q.wfb,Yt,6,2*t.a.length+2*l,15,1),o=0;o<t.a.length;o++)A[o]=t.a[o],A[t.a.length+l+o]=180+t.a[o]<<16>>16;for(e=P;e<i;e++)A[t.a.length+l-1-e+P]=180-t.a[e]<<16>>16,A[A.length-1-e+P]=360-t.a[e]<<16>>16;return A;default:return t.a;}},q.gz=function(t){var A,e,r,P,i,D,u,o,l,U;switch(A=null,U=t.c.length,t.e){case 1:for(A=B(Q.wfb,Yt,6,2*U,15,1),r=0;r<U;r++)A[r]=t.c[r],A[U+r]=t.c[r];return A;case 2:for(P=t.a[0]==0?1:0,i=t.a[U-1]==180?U-1:U,l=i-P,A=B(Q.wfb,Yt,6,U+l,15,1),D=0;D<U;D++)A[D]=t.c[D];for(u=P;u<i;u++)A[A.length-1-u+P]=t.c[u];return A;case 3:for(P=t.a[0]==0?1:0,i=t.a[U-1]==90?U-1:U,l=i-P,A=B(Q.wfb,Yt,6,2*U+2*l,15,1),o=0;o<U;o++)A[o]=t.c[o],A[U+l+o]=t.c[o];for(e=P;e<i;e++)A[U+l-1-e+P]=t.c[e],A[A.length-1-e+P]=t.c[e];return A;default:return t.c;}},q.hz=function(t){var A,e,r,P,i,D,u,o,l,U;switch(A=null,U=t.d.length,t.e){case 1:for(A=T1(Q.wfb,[P0,Yt],[16,6],15,[2*U,2],2),r=0;r<U;r++)A[r][0]=t.d[r][0],A[r][1]=t.d[r][1],A[U+r][0]=180+t.d[r][0]<<16>>16,A[U+r][1]=180+t.d[r][1]<<16>>16;return A;case 2:for(P=t.a[0]==0?1:0,i=t.a[U-1]==180?U-1:U,l=i-P,A=T1(Q.wfb,[P0,Yt],[16,6],15,[U+l,2],2),D=0;D<U;D++)A[D][0]=t.d[D][0],A[D][1]=t.d[D][1];for(u=P;u<i;u++)A[A.length-1-u+P][0]=360-t.d[u][1]<<16>>16,A[A.length-1-u+P][1]=360-t.d[u][0]<<16>>16;return A;case 3:for(P=t.a[0]==0?1:0,i=t.a[U-1]==90?U-1:U,l=i-P,A=T1(Q.wfb,[P0,Yt],[16,6],15,[2*U+2*l,2],2),o=0;o<U;o++)A[o][0]=t.d[o][0],A[o][1]=t.d[o][1],A[U+l+o][0]=180+t.d[o][0]<<16>>16,A[U+l+o][1]=180+t.d[o][1]<<16>>16;for(e=P;e<i;e++)A[U+l-1-e+P][0]=180-t.d[e][1]<<16>>16,A[U+l-1-e+P][1]=180-t.d[e][0]<<16>>16,A[A.length-1-e+P][0]=360-t.d[e][1]<<16>>16,A[A.length-1-e+P][1]=360-t.d[e][0]<<16>>16;return A;default:return t.d;}},q.iz=function(t,A,e,r,P,i){var D,u,o,l,U,y;for(t.a[e]=A<<16>>16,t.c[e]=t.c[e]+t.c[r]<<16>>16,t.d[e][0]=P<<16>>16,t.d[e][1]=i<<16>>16,y=t.a.length-1,o=B(Q.wfb,Yt,6,y,15,1),l=B(Q.wfb,Yt,6,y,15,1),U=B(Q.wfb,P0,16,y,0,2),u=0,D=0;D<t.a.length;D++)D!=r&&(o[u]=t.a[D],l[u]=t.c[D],U[u]=t.d[D],++u);t.a=o,t.c=l,t.d=U;},q.jz=function(t){var A,e,r,P,i,D,u,o,l,U,y,T;for(A=0;t.a.length!=0;){if(t.a.length==1){if(t.e==1||t.e==0)break;if(t.e==2&&(t.a[0]==0||t.a[0]==180))break;if(t.e==3&&(t.a[0]==0||t.a[0]==90))break;}for(T=t.a[0]!=0&&(t.e==2||t.e==3)?-1:0,y=-1,U=Ot,D=T;D<t.a.length;D++)o=D+1,e=D>=0?t.a[D]:-t.a[0],r=o<t.a.length?t.a[o]:t.e==1?t.a[0]+180:t.e==2?360-t.a[t.a[D]==180?D-1:D]:t.e==3?180-t.a[t.a[D]==90?D-1:D]:t.a[0]+360,U>r-e&&(U=r-e,y=D);if(U>15)break;i=y,u=i+1,i==-1?(A+=t.c[0],t.a[0]=0,t.c[0]=t.c[0]*2<<16>>16,t.d[0][0]=-t.d[0][1]):u<t.a.length?(l=q.kz(t,i,u,t.a[u]),q.iz(t,l,i,u,t.d[i][0],t.d[u][1])):t.e==1?(u=0,l=q.kz(t,i,u,t.a[u]+180),l<=180?q.iz(t,l,i,u,t.d[i][0],t.d[u][1]+180):q.iz(t,l-180,u,i,t.d[u][0],t.d[i][1]-180)):t.e==2?(u=t.a[i]==180?i-1:i,t.a[i]<180?(A+=t.c[i],t.a[i]=180,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=360-t.d[i][0]<<16>>16):(h.Rkb(),l=q.kz(t,i,u,t.a[u]),q.iz(t,l,u,i,t.d[u][0],t.d[i][1]))):t.e==3?(u=t.a[i]==90?i-1:i,t.a[i]<90?(A+=t.c[i],t.a[i]=90,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=180-t.d[i][0]<<16>>16):(h.Rkb(),l=q.kz(t,i,u,t.a[u]),q.iz(t,l,u,i,t.d[u][0],t.d[i][1]))):(u=0,l=q.kz(t,i,u,t.a[u]+360),l<=360?q.iz(t,l,i,u,t.d[i][0],t.d[u][1]+360):q.iz(t,l-360,u,i,t.d[u][0],t.d[i][1]-360));}if(A!=0)for(P=0;P<t.c.length;P++)t.c[P]=pt(d.Math.round(100*t.c[P]/(100+A)))<<16>>16;},q.kz=function(t,A,e,r){return pt(d.Math.round((t.a[A]*t.c[A]+r*t.c[e])/(t.c[A]+t.c[e])));},q.lz=function(t){this.e=t;},q.mz=function(t){var A;if(this.e=t.e,t.a!=null)for(this.a=B(Q.wfb,Yt,6,t.a.length,15,1),A=0;A<this.a.length;A++)this.a[A]=360-t.a[this.a.length-A-1]<<16>>16;if(t.d!=null)for(this.d=T1(Q.wfb,[P0,Yt],[16,6],15,[t.d.length,2],2),A=0;A<this.d.length;A++)this.d[A][0]=360-t.d[this.a.length-A-1][1]<<16>>16,this.d[A][1]=360-t.d[this.a.length-A-1][0]<<16>>16;if(t.c!=null)for(this.c=B(Q.wfb,Yt,6,t.c.length,15,1),A=0;A<this.c.length;A++)this.c[A]=t.c[this.c.length-A-1];if(t.b!=null)for(this.b=B(Q.O7,Y,6,t.b.length,15,1),A=0;A<this.b.length;A++)this.b[A]=t.b[this.b.length-A-1];},I(194,1,{194:1},q.lz,q.mz),L.e=0,Q.c9=N(194),q.nz=function(t,A){var e,r,P,i;e=A[1],r=A[2],P=t.j[e]-1,i=t.j[r]-1,P>3||i>3?(this.b=B(Q.wfb,Yt,6,4,15,1),this.b[0]=45,this.b[1]=135,this.b[2]=225,this.b[2]=315,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[4,2],2),this.c[0][0]=30,this.c[0][1]=60,this.c[1][0]=120,this.c[1][1]=150,this.c[2][0]=210,this.c[2][1]=240,this.c[2][0]=300,this.c[2][1]=330,this.a=B(Q.wfb,Yt,6,4,15,1),this.a[0]=25,this.a[1]=25,this.a[2]=25,this.a[3]=25):(t.o[e]==0||t.C[e]>9)&&(t.o[r]==0||t.C[r]>9)?P==3&&i==3||P==3&&i==2||P==3&&i==1||P==2&&i==3||P==1&&i==3||P==2&&i==2&&(A[0]!=-1||A[3]!=-1)?(this.b=B(Q.wfb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=B(Q.wfb,Yt,6,3,15,1),this.a[0]=33,this.a[1]=33,this.a[2]=33):P==1&&i==2&&A[3]==-1||P==2&&i==1&&A[0]==-1?(this.b=B(Q.wfb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=B(Q.wfb,Yt,6,3,15,1),this.a[0]=40,this.a[1]=20,this.a[2]=40):(P==1&&i==1||P==1&&i==2&&A[3]!=-1||P==2&&i==1&&A[0]!=-1||P==2&&i==2&&A[0]==-1&&A[3]==-1)&&(this.b=B(Q.wfb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=B(Q.wfb,Yt,6,3,15,1),this.a[0]=25,this.a[1]=50,this.a[2]=25):(t.o[e]==0||t.C[e]>9)&&t.o[r]==1||(t.o[r]==0||t.C[r]>9)&&t.o[e]==1?P==3||i==3?(this.b=B(Q.wfb,Yt,6,6,15,1),this.b[0]=0,this.b[1]=60,this.b[2]=120,this.b[3]=180,this.b[4]=240,this.b[5]=300,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=45,this.c[1][1]=75,this.c[2][0]=105,this.c[2][1]=135,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=225,this.c[4][1]=255,this.c[5][0]=285,this.c[5][1]=315,this.a=B(Q.wfb,Yt,6,6,15,1),this.a[0]=16,this.a[1]=16,this.a[2]=16,this.a[3]=16,this.a[4]=16,this.a[5]=16):P==1&&i==1?(this.b=B(Q.wfb,Yt,6,2,15,1),this.b[0]=120,this.b[1]=240,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=105,this.c[0][1]=135,this.c[1][0]=225,this.c[1][1]=255,this.a=B(Q.wfb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[e]==1&&P==2&&i==1||t.o[r]==1&&i==2&&P==1?(this.b=B(Q.wfb,Yt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=B(Q.wfb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[e]==1&&P==1&&i==2&&A[3]==-1||t.o[r]==1&&i==1&&P==2&&A[0]==-1?(this.b=B(Q.wfb,Yt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=B(Q.wfb,Yt,6,3,15,1),this.a[0]=60,this.a[1]=20,this.a[2]=20):t.o[e]==1&&P==1&&i==2&&A[3]!=-1||t.o[r]==1&&i==1&&P==2&&A[0]!=-1?(this.b=B(Q.wfb,Yt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=B(Q.wfb,Yt,6,3,15,1),this.a[0]=20,this.a[1]=40,this.a[2]=40):P==2&&i==2&&(A[0]==-1||A[3]==-1?(this.b=B(Q.wfb,Yt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=B(Q.wfb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):(this.b=B(Q.wfb,Yt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=B(Q.wfb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50)):t.o[e]==1&&t.o[r]==1?P==1&&i==1?(this.b=B(Q.wfb,Yt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=B(Q.wfb,Yt,6,2,15,1),this.a[0]=10,this.a[1]=90):(this.b=B(Q.wfb,Yt,6,6,15,1),this.b[0]=0,this.b[1]=50,this.b[2]=130,this.b[3]=180,this.b[4]=230,this.b[5]=310,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=35,this.c[1][1]=65,this.c[2][0]=115,this.c[2][1]=145,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=215,this.c[4][1]=245,this.c[5][0]=295,this.c[5][1]=325,this.a=B(Q.wfb,Yt,6,6,15,1),this.a[0]=40,this.a[1]=5,this.a[2]=5,this.a[3]=40,this.a[4]=5,this.a[5]=5):(this.b=B(Q.wfb,Yt,6,1,15,1),this.b[0]=180,this.c=T1(Q.wfb,[P0,Yt],[16,6],15,[1,2],2),this.c[0][0]=165,this.c[0][1]=195,this.a=B(Q.wfb,Yt,6,1,15,1),this.a[0]=100);},I(251,1,{},q.nz),Q.d9=N(251),q.qz=function(){q.qz=i1,q.pz=j(v(Q.R7,1),w0,6,15,[1,Te,kP,2.119999885559082,1.9800000190734863,1.909999966621399,1.7699999809265137,lr,1.5,dP,1.5800000429153442,2.5,2.509999990463257,2.25,2.190000057220459,mP,1.8899999856948853,Ti,1.8300000429153442,2.7300000190734863,2.619999885559082,2.5799999237060547,hi,ci,Ur,Ur,FP,bi,bi,2.380000114440918,2.390000104904175,vP,jP,1.8799999952316284,Ti,wi,2.25,3.2100000381469727,2.8399999141693115,2.75,2.5199999809265137,2.559999942779541,Ur,FP,hi,FP,2.1500000953674316,gi,Bi,2.430000066757202,ci,ki,1.9900000095367432,2.0399999618530273,2.059999942779541,3.4800000190734863,3.0299999713897705,2.9800000190734863,di,mi,2.950000047683716,mi,2.9000000953674316,Fi,RP,vi,Fi,ji,RP,vi,Ri,2.740000009536743,2.630000114440918,gi,2.569999933242798,Bi,2.4800000190734863,2.4100000858306885,jP,vP,Ur,ki,2.5999999046325684,2.5399999618530273,2.5,2.5,2.5,2.5,2.5,Ri,2.930000066757202,di,2.7100000381469727,2.819999933242798,ji,RP,pi,3.4000000953674316,pi,2.700000047683716]),q.oz=j(v(Q.R7,1),w0,6,15,[.25,pP,.46000000834465027,Ci,1.0199999809265137,he,.75,.7099999785423279,.6299999952316284,.6399999856948853,.9599999785423279,Mi,Si,1.2599999904632568,CP,Hi,1.0299999713897705,.9900000095367432,Ei,Oi,xi,1.4800000190734863,MP,qi,Vi,1.190000057220459,CP,Hi,1.100000023841858,Te,Te,Ii,Lr,Lr,CP,1.1399999856948853,Lr,Ni,SP,HP,1.5399999618530273,Gi,1.3799999952316284,Ki,1.25,1.25,Te,Si,zi,dP,se,se,MP,Ci,1.350000023841858,vP,Oi,or,HP,Ji,1.7400000095367432,Zi,Yi,sr,Xi,sr,_i,lr,EP,1.6399999856948853,ce,Wi,1.5199999809265137,dP,$i,1.309999942779541,ai,Vi,1.2300000190734863,Ii,1.4199999570846558,1.5,zi,1.5099999904632568,t9,Gi,t9,2.2300000190734863,2.009999990463257,wi,1.75,Xi,ce,xi,Yi,lr,lr,sr,sr,EP,_i,Zi,Ji,1.6100000143051147,A9,1.4900000095367432,kP,1.409999966621399,qi,ai,Ki,Lr,$i,MP,kP,Wi,1.75,EP,A9]);},Xt.sz=function(){Xt.sz=i1,Xt.rz=n.po("X",32);},Xt.tz=function(t,A){Xt.sz();var e,r,P,i,D;for(P=0;P<t.q;P++)for(D=n.fw(t),n.Cn(D,P,Xt.rz),n.bw(D,15),r=0;r<D.f;r++)(D.u[r]&O0)!=0&&n.Ip(D,r)==-1&&(i=(n.bw(D,7),D.o[r]==2&&D.j[r]==2?n.nq(D,r,false):n.pq(D,r,false)),i!=-1&&(t.H[i]=257,t.T=0,t.D[1][i]==r&&(e=t.D[0][i],t.D[0][i]=r,t.T=0,t.D[1][i]=e,t.T=0),n.on(t,r,A,0)));},Xt.uz=function(t){Xt.sz();var A,e,r,P;for(Xt.tz(t,1),r=t.q,e=B(Q.Qcb,yt,2,r,6,1),A=0;A<r;A++)P=n.fw(t),n.mn(P,A,(n.Rk(),n.Lk)[P.C[A]]+"*"),P.C[A]==1?n.Cn(P,A,Xt.rz):n.tn(P,A,P.A[A]+5),Xt.vz(P),e[A]=n.rg(new n.Og(P,8));return e;},Xt.vz=function(t){var A;for(n.bw(t,15),A=0;A<t.q;A++)(t.u[A]&3)!=0&&n.on(t,A,1,0);},Xt.rz=0,Xt.wz=function(t,A,e){var r,P;for(r=T1(Q.Qcb,[b0,yt],[13,2],6,[t.f,A],2),n.bw(t,7),P=0;P<t.f;P++)r[P]=Xt.xz(t,P,A,e);return r;},Xt.xz=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g;for(o=new n.uw(t.f,t.g),U=new s._xb(),c=0,T=0,D=B(Q.xfb,Tt,6,t.f,16,1),i=B(Q.S7,H,6,t.f,15,1),g=0;g<e&&T<t.f;g++){if(T==0)i[0]=A,D[A]=true,T=1;else {for(b=T,l=c;l<T;l++)for(P=i[l],y=0;y<t.j[P];y++)if(u=t.i[P][y],!D[u])switch(r){case 0:D[u]=true,i[b++]=u;break;case 1:Xt.zz(t,P)&&Xt.zz(t,u)||(D[u]=true,i[b++]=u);}c=T,T=b;}n.Co(t,o,D,true,null),s.Yxb(U,n.rg(new n.Og(o,8)));}return s.$xb(U,B(Q.Qcb,yt,2,U.a.a.length,6,1));},Xt.yz=function(t,A,e){var r,P,i,D,u,o;for(u=n.Pq(new n._q(true),t),P=-1,i=0;i<u.q;i++)if(r=u.t==null||u.t[i]==null?null:h.akb((D=u.t[i],O1.Enb(),D)),r!=null&&(o=w.UAb("*").length,h.ckb(w.UAb(r).substr(w.UAb(r).length-o,o),"*"))){P=i;break;}return P>=0?Xt.xz(u,P,A,e):B(Q.Qcb,yt,2,0,6,1);},Xt.zz=function(t,A){return !(t.C[A]!=6||t.s[A]!=0||n.op(t,A)+t.j[A]!=4);},Xt.Kz=function(){Xt.Kz=i1,Xt.Az=d.Math.cos(Sn),Xt.Fz=d.Math.sin(Sn),Xt.Cz=d.Math.cos(OP),Xt.Hz=d.Math.sin(OP),Xt.Ez=d.Math.cos(c0),Xt.Jz=d.Math.sin(c0),Xt.Bz=d.Math.cos(be),Xt.Gz=d.Math.sin(be),Xt.Dz=d.Math.cos(lP),Xt.Iz=d.Math.sin(lP),d.Math.cos(n9),d.Math.sin(n9);},Xt.Lz=function(t){Xt.Kz();var A,e;for(n.bw(t,1),e=t.f,A=0;A<e;A++)Xt.Mz(t,A);},Xt.Mz=function(t,A){Xt.Kz();var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut;switch(G=n.op(t,A),lt=t.J[A].a,ut=t.J[A].b,G){case 1:{if(V=t.j[A],V==0?(U=n.Vo(t,true),c=0):(U=lt-n.Ql(t,t.i[A][0]),c=ut-n.Rl(t,t.i[A][0])),V==1)R=n.Tk(t,lt+Xt.Dz*U+Xt.Iz*c,ut-Xt.Iz*U+Xt.Dz*c,0);else if(V==2)U=lt-.5*(n.Ql(t,t.i[A][0])+n.Ql(t,t.i[A][1])),c=ut-.5*(n.Rl(t,t.i[A][0])+n.Rl(t,t.i[A][1])),R=n.Tk(t,lt+U,ut+c,0);else if(V==3){for(ot=t.i[A][0],F=1;F<3;F++)u=t.k[A][F],(t.H[u]==129||t.H[u]==257)&&(ot=t.i[A][F]);if(e=d.Math.abs(n.oo(n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][0]),n.Rl(t,t.i[A][0])),n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][1]),n.Rl(t,t.i[A][1])))),r=d.Math.abs(n.oo(n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][0]),n.Rl(t,t.i[A][0])),n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][2]),n.Rl(t,t.i[A][2])))),P=d.Math.abs(n.oo(n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][1]),n.Rl(t,t.i[A][1])),n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][2]),n.Rl(t,t.i[A][2])))),At=true,e>r&&e>P?r+P<t1&&(At=false,U=lt-.5*(n.Ql(t,t.i[A][0])+n.Ql(t,t.i[A][1])),c=ut-.5*(n.Rl(t,t.i[A][0])+n.Rl(t,t.i[A][1]))):r>e&&r>P?e+P<t1&&(At=false,U=lt-.5*(n.Ql(t,t.i[A][0])+n.Ql(t,t.i[A][2])),c=ut-.5*(n.Rl(t,t.i[A][0])+n.Rl(t,t.i[A][2]))):e+r<t1&&(At=false,U=lt-.5*(n.Ql(t,t.i[A][1])+n.Ql(t,t.i[A][2])),c=ut-.5*(n.Rl(t,t.i[A][1])+n.Rl(t,t.i[A][2]))),At){for(K=t.i[A][0],l=L0,k=0;k<3;k++)i=t.i[A][k],i!=ot&&(o=d.Math.pow(t.J[A].a-t.J[i].a,2)+d.Math.pow(t.J[A].b-t.J[i].b,2),o<l&&(K=i,l=o,h.Rkb()));R=n.Tk(t,(t.J[ot].a+t.J[K].a)/2,(t.J[ot].b+t.J[K].b)/2,0);}else R=n.Tk(t,lt+U,ut+c,0);}else R=n.Tk(t,lt+U,ut+c,0);n.Cn(t,R,1),n.Wk(t,A,R,1);}break;case 2:if(J=t.j[A],J==1)U=lt-n.Ql(t,t.i[A][0]),c=ut-n.Rl(t,t.i[A][0]),R=n.Tk(t,lt+(Xt.Ez*U-Xt.Jz*c)*.7,ut+(Xt.Jz*U+Xt.Ez*c)*.7,0),n.Cn(t,R,1),n.Wk(t,A,R,1),R=n.Tk(t,lt+(Xt.Bz*U-Xt.Gz*c)*.7,ut+(Xt.Gz*U+Xt.Bz*c)*.7,0),n.Cn(t,R,1),n.Wk(t,A,R,1);else if(J==2)y=lt-n.Ql(t,t.i[A][0]),b=ut-n.Rl(t,t.i[A][0]),T=lt-n.Ql(t,t.i[A][1]),g=ut-n.Rl(t,t.i[A][1]),O=d.Math.sqrt(y*y+b*b)*.7,x=d.Math.sqrt(T*T+g*g)*.7,U=y+T,c=b+g,E=d.Math.sqrt(U*U+c*c),D=(O+x)/2,U=U/E*D,c=c/E*D,it=n.Ip(t,A),R=n.Tk(t,lt+Xt.Az*U-Xt.Fz*c,ut+Xt.Fz*U+Xt.Az*c,0),n.Cn(t,R,1),it>-1?n.Wk(t,A,R,1):n.Wk(t,A,R,257),R=n.Tk(t,lt+Xt.Cz*U-Xt.Hz*c,ut+Xt.Hz*U+Xt.Cz*c,0),n.Cn(t,R,1),n.Wk(t,A,R,1);else for(S=0;S<G;S++)R=n.Tk(t,lt,ut,0),n.Cn(t,R,1),n.Wk(t,A,R,1);break;case 3:t.i[A][0]>0?(U=(lt-n.Ql(t,t.i[A][0]))*.7,c=(ut-n.Rl(t,t.i[A][0]))*.7,R=n.Tk(t,lt+U,ut+c,0),n.Cn(t,R,1),n.Wk(t,A,R,1),R=n.Tk(t,lt-c,ut+U,0),n.Cn(t,R,1),n.Wk(t,A,R,1),R=n.Tk(t,lt+c,ut-U,0),n.Cn(t,R,1),n.Wk(t,A,R,1)):(U=n.Vo(t,true),c=n.Vo(t,true),R=n.Tk(t,lt+U,ut+c,0),n.Cn(t,R,1),n.Wk(t,A,R,1),R=n.Tk(t,lt-c*Xt.Ez,ut+U*Xt.Jz,0),n.Cn(t,R,1),n.Wk(t,A,R,1),R=n.Tk(t,lt-c*Xt.Ez,ut-U*Xt.Jz,0),n.Cn(t,R,1),n.Wk(t,A,R,1));break;default:{for(S=0;S<G;S++)R=n.Tk(t,lt,ut,0),n.Cn(t,R,1),n.Wk(t,A,R,1);break;}}},Xt.Nz=function(t){Xt.Kz();var A,e;for(n.bw(t,1),e=0,A=0;A<t.q;A++)t.C[A]==1?++e:e+=t.e[A]-t.j[A]+n.op(t,A);return e;},Xt.Az=0,Xt.Bz=0,Xt.Cz=0,Xt.Dz=0,Xt.Ez=0,Xt.Fz=0,Xt.Gz=0,Xt.Hz=0,Xt.Iz=0,Xt.Jz=0,X.Pz=function(t,A,e){var r,P,i;for(i=A.length,r=new X.uB(t.f,i,false),r.a[0]=0,r.b[0]=0,P=0;P<i;P++)r.k[P]=128-i,r.e[P]=A[P];i<8?X.Yz(r):X.Wz(t,r,A,e),X.FA(t.d,r);},X.Qz=function(t,A,e,r,P){var i,D,u,o,l,U,y,T;for(D=B(Q.S7,H,6,P,15,1),i=0;i<t.f.q;i++){for(o=0;o<n.Lo(t.f,i);o++){for(T=e[n.Zo(t.f,i,o)],l=0;l<o&&!(T<D[l]);l++);for(U=o;U>l;U--)D[U]=D[U-1];D[l]=T;}for(y=n.Lo(t.f,i),n.eh(A[i],i),n.ah(A[i],r,Q.Mfb(e[i])),n.ah(A[i],(P-y)*(r+1),0),u=0;u<y;u++)n.ah(A[i],r+1,Q.Mfb(D[u]));}},X.Rz=function(t){var A,e,r,P,i,D,u,o,l,U,y;for(r=n.Pg(t.f.f),o=2,e=0;e<t.f.f;e++)o=d.Math.max(o,n.Lo(t.f,e));for(i=(62+2*r+o*(r+1))/63|0,P=B(Q.c8,Hn,101,t.f.q,0,1),A=0;A<t.f.q;A++)P[A]=new n.fh(i),n.eh(P[A],A);for(y=B(Q.S7,H,6,t.f.q,15,1),D=0;D<t.f.g;D++)u=n.gm(t.f,D),(u==1||u==2)&&(n.bh(P[n.Zl(t.f,0,D)],Q.Mfb(u)),n.bh(P[n.Zl(t.f,1,D)],Q.Mfb(u)));l=X.Sz(P,y);do U=l,X.Qz(t,P,y,r,o),l=X.Sz(P,y);while(U!=l);return y;},X.Sz=function(t,A){var e,r;for(r=0,s.Trb(t,0,t.length,null),e=0;e<t.length;e++)(e==0||n.dh(t[e],t[e-1])!=0)&&++r,A[t[e].a]=r;return r;},X.Tz=function(t){var A,e,r,P,i,D,u,o,l,U,y,T;for(o=new s.trb(t.d);o.a<o.c.a.length;)for(u=s.srb(o),l=0;l<u.f.length;l++)if(r=u.f[l],n.fm(t.f,r)==2&&(!n.jq(t.f,r)&&(n.gm(t.f,r)==3||n.gm(t.f,r)==0)&&n.Kn(t.f,r),!n.gq(t.f,r)&&n.$o(t.f,n.Zl(t.f,0,r))>1&&n.$o(t.f,n.Zl(t.f,1,r))>1&&(n.gm(t.f,r)==1||n.gm(t.f,r)==2))){for(T=B(Q.S7,H,6,2,15,1),P=B(Q.S7,H,6,2,15,1),U=0;U<2;U++)for(T[U]=t.f.M,P[U]=n.Zl(t.f,U,r),y=0;y<n.Lo(t.f,P[U]);y++)i=n.Zo(t.f,P[U],y),i!=n.Zl(t.f,1-U,r)&&T[U]>i&&(T[U]=i);D=X.$A(u.a[u.g[P[0]]],u.b[u.g[P[0]]],u.a[u.g[P[1]]],u.b[u.g[P[1]]]),A=X.$A(u.a[u.g[T[0]]],u.b[u.g[T[0]]],u.a[u.g[P[0]]],u.b[u.g[P[0]]]),e=X.$A(u.a[u.g[P[1]]],u.b[u.g[P[1]]],u.a[u.g[T[1]]],u.b[u.g[T[1]]]),X.$z(D,A)<0^X.$z(D,e)<0^n.gm(t.f,r)==2&&X.iB(u,r);}},X.Uz=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F;for(T=new s.Mj(),F=new s.Mj(),r=new s.Mj(),i=new s.trb(t.d);i.a<i.c.a.length;){for(P=s.srb(i),D=0,w.mAb(r.a,0),u=0;u<P.e.length;u++)A=P.e[u],e=t.i[A],e!=0&&(s.sj(r,new X.aB(P,u,e)),D+=e);if(D!=0)for(s.Jj(r,new X.RA()),l=new s.trb(r);l.a<l.c.a.length&&(o=s.srb(l),!(D*o.b>0&&(e=d.Math.abs(D)>=d.Math.abs(o.b)?o.b:D,D-=e,s.sj(e<0?T:F,new X.aB(P,o.a,e)),D==0))););}if(T.a.length==0||F.a.length==0)return null;for(s.Jj(F,new X.KA()),s.Jj(T,new X.OA()),g=new s.trb(F);g.a<g.c.a.length;)for(c=s.srb(g),y=new s.trb(T);y.a<y.c.a.length;)if(U=s.srb(y),c.b==-U.b)return t.i[X.mB(c.c,c.a)]-=c.b,t.i[X.mB(U.c,U.a)]-=U.b,new X.XA(c.c,U.c,c.a,U.a);for(k=new s.trb(F);k.a<k.c.a.length;)for(c=s.srb(k),y=new s.trb(T);y.a<y.c.a.length;)if(U=s.srb(y),c.b>-U.b)return t.i[X.mB(c.c,c.a)]+=U.b,t.i[X.mB(U.c,U.a)]-=U.b,new X.XA(c.c,U.c,c.a,U.a);for(b=new s.trb(F);b.a<b.c.a.length;)for(c=s.srb(b),y=new s.trb(T);y.a<y.c.a.length;)if(U=s.srb(y),c.b<-U.b)return t.i[X.mB(c.c,c.a)]-=c.b,t.i[X.mB(U.c,U.a)]+=c.b,new X.XA(c.c,U.c,c.a,U.a);return null;},X.Vz=function(t){return t.d.a.length<2?null:new X.WA(s.xj(t.d,0),s.xj(t.d,1));},X.Wz=function(t,A,e,r){var P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V;if(U=j(v(Q.Q7,2),Q1,15,0,[j(v(Q.Q7,1),jt,6,15,[20]),null,null,j(v(Q.Q7,1),jt,6,15,[0,10]),null,null,j(v(Q.Q7,1),jt,6,15,[-4,12]),j(v(Q.Q7,1),jt,6,15,[0,0,-7.5]),null,null,null,null,j(v(Q.Q7,1),jt,6,15,[8.571428571428571,-8.571428571428571]),null,null,null,j(v(Q.Q7,1),jt,6,15,[-2.4])]),y=j(v(Q.S7,2),E1,7,0,[j(v(Q.S7,1),H,6,15,[146]),j(v(Q.S7,1),H,6,15,[627]),null,j(v(Q.S7,1),H,6,15,[2457,1170]),null,j(v(Q.S7,1),H,6,15,[2451,8643,2519]),j(v(Q.S7,1),H,6,15,[9362,14798]),j(v(Q.S7,1),H,6,15,[34377,-2147448999,26214]),null,j(v(Q.S7,1),H,6,15,[37449,137313,95703,34371,37815,54891,132867,-2147309741,54857,55129,-2147449005,-2147449065]),null,j(v(Q.S7,1),H,6,15,[530697,531819,899169,137289,694617,-2146951863,-2146952797,-2146939175,-2146929547,-2146929564,-2146625111,-2146931799,-2146940503,-2146931935]),j(v(Q.S7,1),H,6,15,[1007293,610915]),j(v(Q.S7,1),H,6,15,[542985,137283,2122017,530691,2206773,-2144711351,219209,2840841,137555,-2146871031,-2147264167,613705,-2145360543,-2146625271,694611,2454837,-2145356703,-2147345133,-2146928951,-2146931805,-2144641719,-2146951869,-2146625237,-2146624183,2841963,1074905,-2146625117,2799955,-2144723645,138583,859225,-2145264843,-2145216253,-2146624149,-2144700727,-2146928917,-2143905527,-2144045771,-2146789097,2288547,544407,2104323,-2146911977,-2144479405,3633737,-2146870089,-2146952169]),null,j(v(Q.S7,1),H,6,15,[8487297,2172633,2116611,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8829813]),j(v(Q.S7,1),H,6,15,[14071213])]),S=1<<A.e.length,D=0,o=0,A.e.length>7)for(c=0;c<A.e.length;c++)u=X.cA(t,e,r,c),u==1?D+=S:u==2&&(o+=S),D>>>=1,o>>>=1;if(O=A.e.length-9,A.e.length>=9&&A.e.length<=25&&y[O]!=null)for(V=0;V<y[O].length;V++)for(F=(e1&y[O][V])==0,l=Ot&y[O][V],g=false;!g;g=!g){if(g){if(F)break;for(R=0,P=1;P!=S;P<<=1)R<<=1,(l&P)!=0&&(R|=1);l=R;}for(x=0;x<A.e.length;x++){if((l&D)==0&&(~l&o)==0){for(i=0,T=e9*(U[O]==null?0:U[O][V]),E=0,G=l,k=true,b=0;b<A.e.length;b++)k&&++E,(G&1)==0&&(k=!k),G>>>=1;for(J=E>(A.e.length/2|0),c=1;c<A.e.length;c++)A.a[c]=A.a[c-1]+d.Math.sin(i),A.b[c]=A.b[c-1]+d.Math.cos(i),(l&1)==0&&(J=!J),i+=T+(J?c0:be),l>>>=1;return;}(l&1)!=0&&(l|=S),l>>>=1;}}X.Zz(A,D,o);},X.Xz=function(t){var A,e,r,P,i,D,u,o,l,U;for(A=null,l=null,D=0;D<t.f.g;D++)if(n.im(t.f,D)==16){for(e=n.Zl(t.f,0,D),P=-1,u=0;u<t.d.a.length&&(P=X.nB(s.xj(t.d,u),e),P==-1);u++);for(r=n.Zl(t.f,1,D),i=-1,o=0;o<t.d.a.length&&(i=X.nB(s.xj(t.d,o),r),i==-1);o++);u!=o&&(u>o&&(U=u,u=o,o=U,U=P,P=i,i=U),l==null&&(l=B(Q.j9,b0,461,t.d.a.length,0,2)),l[o]==null&&(l[o]=B(Q.j9,{461:1,4:1,5:1,8:1},106,o,0,1)),l[o][u]?X.UA(l[o][u],P,i):(a1(l[o],u,new X.XA(s.xj(t.d,u),s.xj(t.d,o),P,i)),!A&&(A=new s.Mj()),s.sj(A,l[o][u])));}return A;},X.Yz=function(t){var A,e;for(A=t1-t1*(t.e.length-2)/t.e.length,e=1;e<t.e.length;e++)t.a[e]=t.a[e-1]+d.Math.sin(A*(e-1)),t.b[e]=t.b[e-1]+d.Math.cos(A*(e-1));},X.Zz=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F;if(A==0||(A&e)!=0){X.Yz(t);return;}for(k=-1,F=0,i=1<<t.e.length-2,P=1<<t.e.length-1,u=1,D=2,l=0;l<t.e.length;l++)(e&(P|u))==0&&(A&(P|u))!=0&&(A&i)==0&&(b=0,(e&i)!=0&&(b+=4),(A&P)!=0&&(b+=2),(A&u)!=0&&(b+=1),F<b&&(F=b,k=l)),i=P,P=u,u=D,D=1<<(l+2<t.e.length?l+2:l+2-t.e.length);if(k==-1){X.Yz(t);return;}for(T=0,T|=1<<k,c=2;c<t.e.length-1;){if(y=k+c<t.e.length?k+c:k+c-t.e.length,P=1<<(y==0?t.e.length-1:y-1),(e&P)!=0){++c;continue;}if(u=1<<y,(A&P)!=0){if((e&u)!=0){X.Yz(t);return;}T|=u,c+=2;continue;}if(D=1<<(y+1<t.e.length?y+1:y+1-t.e.length),(A&u)!=0&&(e&D)!=0){T|=u,c+=3;continue;}++c;}if(T==0){X.Yz(t);return;}for(r=t1-t1*(t.e.length-2)/t.e.length,U=1;U<t.e.length;U++)t.a[U]=t.a[U-1]+d.Math.sin(r*(U-1)),t.b[U]=t.b[U-1]+d.Math.cos(r*(U-1));for(u=1,g=2*d.Math.sin(r/2),o=0;o<t.e.length;o++)(T&u)!=0&&(t.a[o]+=g*d.Math.cos(r*(o-.5)),t.b[o]-=g*d.Math.sin(r*(o-.5))),u<<=1;},X.$z=function(t,A){var e;for(e=t-A;e<QP;)e+=N1;for(;e>t1;)e-=N1;return e;},X._z=function(t,A,e){var r,P;for(r=0,P=0;P<n.Lo(t.f,e);P++)X.pB(A,n.Zo(t.f,e,P))&&++r;return r;},X.aA=function(t,A,e,r){var P,i,D,u,o;return u=X.nB(A,r),o=X.nB(e,r),X.tB(e,A.a[u]-e.a[o],A.b[u]-e.b[o]),P=X.zA(t,A,r),i=X.zA(t,e,r),D=0,X._z(t,A,r)==1&&X._z(t,e,r)==1&&(D=c0),X.sB(e,e.a[o],e.b[o],P-i+D+t1),X.fA(t,A,e,1);},X.bA=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt;for(R=B(Q.S7,H,6,P,15,1),E=B(Q.S7,H,6,P,15,1),g=0;g<P;g++)R[g]=X.nB(A,r[g]),E[g]=X.nB(e,r[g]);for(ot=0,it=0,K=0,lt=0,k=0;k<P;k++)ot+=A.a[R[k]],it+=A.b[R[k]],K+=e.a[E[k]],lt+=e.b[E[k]];for(ot/=P,it/=P,K/=P,lt/=P,X.tB(e,ot-K,it-lt),l=B(Q.k9,C1,41,P,0,1),y=B(Q.k9,C1,41,P,0,1),i=B(Q.k9,C1,41,P,0,1),D=B(Q.k9,C1,41,P,0,1),F=0;F<P;F++)l[F]=new X.ZA(ot,it,A.a[R[F]],A.b[R[F]]),y[F]=new X.ZA(ot,it,e.a[E[F]],e.b[E[F]]),i[F]=new X.YA(l[F].a-y[F].a,l[F].b*y[F].b),D[F]=new X.YA(l[F].a+y[F].a,l[F].b*y[F].b);for(x=X.DA(i,P),G=X.DA(D,P),Dt=0,Qt=0,S=0;S<P;S++)for(O=0;O<n.Lo(t.f,r[S]);O++)u=n.Zo(t.f,r[S],O),X.pB(A,u)&&!X.pB(e,u)&&++Dt,!X.pB(A,u)&&X.pB(e,u)&&++Qt;for(U=B(Q.k9,C1,41,Dt,0,1),T=B(Q.k9,C1,41,Qt,0,1),c=B(Q.k9,C1,41,Qt,0,1),Dt=0,Qt=0,b=0;b<P;b++)for(O=0;O<n.Lo(t.f,r[b]);O++)u=n.Zo(t.f,r[b],O),X.pB(A,u)&&!X.pB(e,u)&&(o=X.nB(A,u),U[Dt]=new X.ZA(A.a[R[b]],A.b[R[b]],A.a[o],A.b[o]),++Dt),!X.pB(A,u)&&X.pB(e,u)&&(o=X.nB(e,u),ut=new X.ZA(e.a[E[b]],e.b[E[b]],e.a[o],e.b[o]),T[Qt]=new X.YA(x.a+ut.a,ut.b),c[Qt]=new X.YA(G.a-ut.a,ut.b),++Qt);return J=X.DA(U,Dt),V=X.DA(T,Qt),At=X.DA(c,Qt),d.Math.abs(X.$z(J.a,V.a))>d.Math.abs(X.$z(J.a,At.a))?X.sB(e,ot,it,x.a):(X.hB(e,ot,it),X.sB(e,ot,it,G.a)),X.fA(t,A,e,P);},X.cA=function(t,A,e,r){var P,i,D,u,o,l;return i=r==A.length-1?0:r+1,u=r==0?A.length-1:r-1,D=i==A.length-1?0:i+1,n.fm(t.f,e[r])==2&&(P=n.gm(t.f,e[r]),P==1||P==2)?(X.kA(t,A[u],A[r],A[i])^X.kA(t,A[D],A[i],A[r])&&(P=P==1?2:1),P):n.jq(t.f,e[r])?(o=n.lt(n.Ep(t.f),e[u],e[r]),l=n.lt(n.Ep(t.f),e[i],e[r]),o!=-1||l!=-1?o==l?2:1:2):0;},X.dA=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c;for(P=B(Q.S7,H,6,t.f.q,15,1),i=B(Q.S7,H,6,t.f.q,15,1),D=B(Q.S7,H,6,t.f.q,15,1),u=B(Q.S7,H,6,t.f.q,15,1),P[0]=A,D[A]=1,u[0]=-1,r=0,o=0;r<=o;){if(r==0||!t.b[P[r]])for(l=0;l<n.Lo(t.f,P[r]);l++)e=n.Zo(t.f,P[r],l),T=n._o(t.f,P[r],l),D[e]==0&&!t.c[T]&&(P[++o]=e,i[o]=T,D[e]=D[P[r]]+1,u[o]=r);if(r==o){for(c=new X._A(D[P[r]]),U=r,y=0;y<c.a.length;y++)c.a[y]=P[U],c.b[y]=i[U],U=u[U];return c;}++r;}return null;},X.eA=function(t){var A,e,r,P;for(P=0,r=null,e=new s.trb(t);e.a<e.c.a.length;)A=s.srb(e),P<A.b[0].e.length*A.b[1].e.length&&(P=A.b[0].e.length*A.b[1].e.length,r=A);return r;},X.fA=function(t,A,e,r){var P,i,D,u,o;for(i=new X.uB(t.f,A.e.length+e.e.length-r,A.i|e.i),P=0,u=0;u<A.e.length;u++)i.e[P]=A.e[u],i.k[P]=A.k[u],i.a[P]=A.a[u],i.b[P++]=A.b[u];for(D=0;D<e.e.length;D++)o=X.nB(A,e.e[D]),o==-1?(i.e[P]=e.e[D],i.k[P]=e.k[D],i.a[P]=e.a[D],i.b[P++]=e.b[D]):i.k[o]<e.k[D]&&(i.k[o]=e.k[D],i.a[o]=e.a[D],i.b[o]=e.b[D]);return i;},X.gA=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b;for(u=B(Q.S7,H,6,t.f.q,15,1),o=B(Q.S7,H,6,t.f.q,15,1),l=B(Q.S7,H,6,t.f.q,15,1),U=B(Q.S7,H,6,t.f.q,15,1),u[0]=e,l[e]=1,U[0]=-1,D=0,y=0;D<=y;){for(T=0;T<n.Lo(t.f,u[D]);T++){if(P=n.Zo(t.f,u[D],T),b=n._o(t.f,u[D],T),P==A){for(i=l[u[D]],r=B(Q.S7,H,6,i,15,1),r[0]=b,c=1;c<i;c++)r[c]=o[D],D=U[D];return r;}l[P]==0&&(u[++y]=P,o[y]=b,l[P]=l[u[D]]+1,U[y]=D);}if(D==y)return null;++D;}return null;},X.hA=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b;for(e=n.Zl(t.f,0,A),r=n.Zl(t.f,1,A),D=B(Q.S7,H,6,t.f.q,15,1),u=B(Q.S7,H,6,t.f.q,15,1),o=B(Q.S7,H,6,t.f.q,15,1),l=B(Q.S7,H,6,t.f.q,15,1),D[0]=e,D[1]=r,u[1]=A,o[e]=1,o[r]=2,l[0]=-1,l[1]=0,i=1,U=1;i<=U;){for(y=0;y<n.$o(t.f,D[i]);y++){if(P=n.Zo(t.f,D[i],y),i>1&&P==e){for(b=new X._A(o[D[i]]),u[0]=n._o(t.f,D[i],y),T=i,c=0;c<b.a.length;c++)b.a[c]=D[T],b.b[c]=u[T],T=l[T];return b;}o[P]==0&&n.fq(t.f,P)&&(D[++U]=P,u[U]=n._o(t.f,D[i],y),o[P]=o[D[i]]+1,l[U]=i);}++i;}return null;},X.iA=function(t,A,e,r){var P,i,D,u,o,l;for(D=B(Q.S7,H,6,t.f.q,15,1),u=B(Q.S7,H,6,t.f.q,15,1),D[0]=e,D[1]=A,u[e]=1,u[A]=2,i=1,o=1;i<=o;){for(l=0;l<n.$o(t.f,D[i]);l++){if(P=n.Zo(t.f,D[i],l),P==r)return 1+u[D[i]];u[P]==0&&n.fq(t.f,P)&&(D[++o]=P,u[P]=u[D[i]]+1);}++i;}return 0;},X.jA=function(t,A){var e,r,P,i,D,u,o;for(u=(A.T&15)!=0,o=A.T&248,!t.g&&(t.g=new s.vwb()),(t.e&2)!=0&&n.qq(A,false),t.f=A,n.bw(t.f,7),t.d=new X.HA(),t.b=B(Q.xfb,Tt,6,t.f.q,16,1),t.c=B(Q.xfb,Tt,6,t.f.r,16,1),t.i=B(Q.S7,H,6,t.f.q,15,1),e=0;e<t.f.q;e++)t.i[e]=n.Bl(t.f,e);for((t.e&12)!=0&&X.uA(t),(t.e&1)==0&&X.Oz&&X.wA(t,X.Oz),X.tA(t),X.pA(t),X.rA(t),X.pA(t),i=new s.trb(t.d);i.a<i.c.a.length;)r=s.srb(i),X.qB(r);for(X.Tz(t),X.xA(t),X.vA(t),X.oA(t),X.mA(t),X.qA(t),P=new s.trb(t.d);P.a<P.c.a.length;)for(r=s.srb(P),D=0;D<r.e.length;D++)n.zn(t.f,r.e[D],r.a[D]),n.An(t.f,r.e[D],r.b[D]),n.Bn(t.f,r.e[D],0);u&&(n.vq(t.f,o),n.yq(t.f)),t.a&&n.Zm(t.f);},X.kA=function(t,A,e,r){var P,i;for(i=0;i<n.$o(t.f,e);i++)if(P=n.Zo(t.f,e,i),P!=r&&P<A)return  false;return  true;},X.lA=function(t,A,e){var r;X.VA(A,e,(t.e&12)!=0),r=X.fA(t,A.b[0],A.b[1],0),X.AA(t,A.b[0],A.b[1],r);},X.mA=function(t){var A;for(A=X.Uz(t);A;)X.lA(t,A,se),A=X.Uz(t);},X.nA=function(t,A,e,r){var P,i,D,u,o;for(P=B(Q.S7,H,6,r,15,1),i=0,D=0;D<A.e.length;D++)for(u=0;u<e.e.length;u++)A.e[D]==e.e[u]&&(P[i++]=A.e[D]);o=r==1?X.aA(t,A,e,P[0]):X.bA(t,A,e,P,r),X.AA(t,A,e,o);},X.oA=function(t){var A,e;for(e=X.Xz(t);e;)A=X.eA(e),X.lA(t,A,Te),e=X.Xz(t);},X.pA=function(t){for(var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S;;){for(S=0,c=0,k=null,F=null,D=1;D<t.d.a.length;D++)for(r=s.xj(t.d,D),u=0;u<D;u++){for(P=s.xj(t.d,u),A=0,e=0,b=0,g=0,U=0;U<r.e.length;U++)for(T=0;T<P.e.length;T++)r.e[U]==P.e[T]&&(++e,A=r.e[U],b<r.k[U]&&(b=r.k[U]),g<P.k[T]&&(g=P.k[T]));if(e>0&&(i=e==1&&X._z(t,r,A)==1&&X._z(t,P,A)==1?0:1,b>g?o=(i<<24)+(b<<16)+(g<<8)+e:o=(i<<24)+(g<<16)+(b<<8)+e,S<o)){for(S=o,c=e,b=0,g=0,y=0;y<r.e.length;y++)b<r.k[y]&&(b=r.k[y]);for(l=0;l<P.e.length;l++)g<P.k[l]&&(g=P.k[l]);b>g?(k=r,F=P):(k=P,F=r);}}if(S==0)break;c==k.e.length?s.Fj(t.d,k):c==F.e.length?s.Fj(t.d,F):X.nA(t,k,F,c);}},X.qA=function(t){var A;for(A=X.Vz(t);A;)X.lA(t,A,Mi),A=X.Vz(t);},X.rA=function(t){for(var A,e,r,P,i,D,u;;){for(i=null,A=0;A<t.f.q;A++){for(u=0,P=0;P<n.Lo(t.f,A);P++)t.c[n._o(t.f,A,P)]||++u;u==1&&(D=X.dA(t,A),(!i||D.a.length>i.a.length)&&(i=D));}if(!i)break;for(e=new X.uB(t.f,i.a.length,false),r=0;r<i.a.length;r++)t.b[i.a[r]]=true,r<i.a.length-1&&(t.c[i.b[r]]=true),e.e[r]=i.a[r],e.a[r]=d.Math.cos(Sn)*r,e.b[r]=(r&1)==1?0:.5,e.k[r]=128+i.a.length;X.FA(t.d,e);}},X.sA=function(t,A,e){var r,P,i,D,u,o,l,U,y,T;for(i=0;i<t.f.r;i++)if(r=n.Zl(t.f,0,i),P=n.Zl(t.f,1,i),!(n.gq(t.f,i)||n.fm(t.f,i)!=1||n.Lo(t.f,r)==1||n.Lo(t.f,P)==1)&&!((t.e&4)!=0&&n.Km(t.f,r)&&n.Km(t.f,P))){for(y=false,l=0;l<2;l++)if(D=n.Zl(t.f,l,i),n.Lo(t.f,D)>2){for(T=true,o=-1,U=0;U<n.Lo(t.f,D);U++)u=n.Zo(t.f,D,U),u!=n.Zl(t.f,1-l,i)&&(o==-1?o=e[u]:o!=e[u]&&(T=false));if(T){y=true;break;}}y||((t.e&8)!=0&&n.Km(t.f,r)&&n.Km(t.f,P)?A[i]=1:A[i]=2);}},X.tA=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K,it,lt,ut,Dt,Qt,bt;for(r=0;r<t.f.f;r++)if(n.Lo(t.f,r)>4){for(T=new X.uB(t.f,1+n.Lo(t.f,r),false),T.a[n.Lo(t.f,r)]=0,T.b[n.Lo(t.f,r)]=0,T.k[n.Lo(t.f,r)]=32,T.e[n.Lo(t.f,r)]=r,t.b[r]=true,b=0;b<n.Lo(t.f,r);b++)l=n.Zo(t.f,r,b),T.a[b]=d.Math.sin(c0*b-qn),T.b[b]=d.Math.cos(c0*b-qn),T.k[b]=32,T.e[b]=l,t.b[l]=true,t.c[n._o(t.f,r,b)]=true;X.FA(t.d,T);}for(lt=n.Ep(t.f),it=0;it<lt.j.a.length;it++){if(ut=s.xj(lt.k,it).length,ot=s.xj(lt.j,it),Dt=false,(t.e&12)!=0){for(Dt=true,b=0;b<ut;b++)if(!n.Km(t.f,ot[b])){Dt=false;break;}}if(!Dt){for(F=false,g=0;g<ut;g++)if(n.Uo(t.f,ot[g])==ut){F=true;break;}if(F)for(K=s.xj(lt.k,it),X.Pz(t,ot,K),b=0;b<ut;b++)t.b[ot[b]]=true,t.c[K[b]]=true;}}for(u=0;u<t.f.g;u++)if(n.gq(t.f,u)&&!t.c[u])for(bt=X.hA(t,u),ot=bt.a,K=bt.b,X.Pz(t,ot,K),b=0;b<bt.a.length;b++)t.b[ot[b]]=true,t.c[K[b]]=true;for(o=0;o<t.f.r;o++)if(!t.c[o]&&n.fm(t.f,o)==3&&(P=n.Zl(t.f,0,o),i=n.Zl(t.f,1,o),x=n.Lo(t.f,P)+n.Lo(t.f,i),x>2)){for(T=new X.uB(t.f,x,false),U=0,g=0;g<n.Lo(t.f,P);g++)l=n.Zo(t.f,P,g),l!=i&&(T.e[U++]=l,t.b[l]=true,t.c[n._o(t.f,P,g)]=true);for(T.e[U++]=P,T.e[U++]=i,k=0;k<n.Lo(t.f,i);k++)l=n.Zo(t.f,i,k),l!=P&&(T.e[U++]=l,t.b[l]=true,t.c[n._o(t.f,i,k)]=true);for(b=0;b<x;b++)T.a[b]=b,T.b[b]=0,T.k[b]=1;t.b[P]=true,t.b[i]=true,t.c[o]=true,X.FA(t.d,T);}for(D=0;D<t.f.r;D++)if(!t.c[D]&&n.fm(t.f,D)==2){for(A=B(Q.S7,H,6,t.f.q,15,1),b=0;b<2;b++)if(A[0]=n.Zl(t.f,b,D),A[1]=n.Zl(t.f,1-b,D),n.Qo(t.f,A[0])==1&&n.Qo(t.f,A[1])==2&&n.Lo(t.f,A[1])==2){t.b[A[0]]=true,t.b[A[1]]=true,t.c[D]=true,O=1;do{if(G=n.Zo(t.f,A[O],0)==A[O-1]?1:0,A[O+1]=n.Zo(t.f,A[O],G),n.Qo(t.f,A[O+1])==2&&n.Lo(t.f,A[O+1])>2)break;t.b[A[O+1]]=true,t.c[n._o(t.f,A[O],G)]=true,++O;}while(n.Qo(t.f,A[O])==2&&n.Lo(t.f,A[O])==2);for(x=n.Lo(t.f,A[0])+n.Lo(t.f,A[O])+O-1,T=new X.uB(t.f,x,false),R=0;R<=O;R++)T.a[R]=R,T.b[R]=0,T.k[R]=64,T.e[R]=A[R];for(y=O+1,c=false,E=0;E<n.Lo(t.f,A[0]);E++)l=n.Zo(t.f,A[0],E),l!=A[1]&&(T.a[y]=-0.5,T.b[y]=c?d.Math.sin(c0):-d.Math.sin(c0),T.k[y]=64,T.e[y]=l,++y,c=true);for(c=false,S=0;S<n.Lo(t.f,A[O]);S++)l=n.Zo(t.f,A[O],S),l!=A[O-1]&&(T.a[y]=O+.5,T.b[y]=c?-d.Math.sin(c0):d.Math.sin(c0),T.k[y]=64,T.e[y]=l,++y,c=true);X.FA(t.d,T);}}for(e=0;e<t.f.q;e++)if(n.Lo(t.f,e)==4){for(J=B(Q.S7,H,6,4,15,1),V=B(Q.S7,H,6,4,15,1),At=0,g=0;g<4;g++)J[At]=n.Zo(t.f,e,g),V[At]=n._o(t.f,e,g),n.Lo(t.f,J[At])==1&&!t.c[V[At]]&&++At;if(At==2){for(T=new X.uB(t.f,3,false),b=0;b<2;b++)t.b[J[b]]=true,t.c[V[b]]=true,T.e[b]=J[b],T.k[b]=32;T.a[0]=-0.5,T.b[0]=.866,T.a[1]=.5,T.b[1]=.866,T.a[2]=0,T.b[2]=0,T.k[2]=32,T.e[2]=e,X.FA(t.d,T);}if(At==3){for(k=0;k<2;k++)n.fm(t.f,V[k])==1&&(Qt=J[k],J[k]=J[2],J[2]=Qt,Qt=V[k],V[k]=V[2],V[2]=Qt);for(T=new X.uB(t.f,4,false),b=0;b<3;b++)t.b[J[b]]=true,t.c[V[b]]=true,T.e[b]=J[b],T.k[b]=32;T.a[0]=-1,T.b[0]=0,T.a[1]=1,T.b[1]=0,T.a[2]=0,T.b[2]=1,T.a[3]=0,T.b[3]=0,T.k[3]=32,T.e[3]=e,X.FA(t.d,T);}}},X.uA=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O;for(u=0,e=0;e<t.f.q;e++)n.Km(t.f,e)&&++u;if(!(u<2)){for(y=0,l=0,U=0;U<t.f.r;U++)r=n.Zl(t.f,0,U),P=n.Zl(t.f,1,U),n.Km(t.f,r)&&n.Km(t.f,P)&&(t.c[U]=true,t.b[r]=true,t.b[P]=true,l+=n.dm(t.f,U),++y);for(y!=0&&l!=0?l/=y:l=n.Ul(t.f),i=0;i<t.f.q;i++)n.Km(t.f,i)&&!t.b[i]&&--u;if(!(u<2)){for(R=B(Q.S7,H,6,t.f.q,15,1),T=n.hp(t.f,R,true,true),S=B(Q.S7,H,6,T,15,1),D=0;D<t.f.q;D++)R[D]!=-1&&++S[R[D]];for(F=B(Q.o9,C1,47,T,0,1),b=0;b<T;b++)F[b]=new X.uB(t.f,S[b],true);for(o=B(Q.S7,H,6,T,15,1),A=0;A<t.f.q;A++)g=R[A],g!=-1&&(F[g].k[o[g]]=f0,F[g].e[o[g]]=A,F[g].a[o[g]]=n.Ql(t.f,A)/l,F[g].b[o[g]]=n.Rl(t.f,A)/l,++o[g]);for(E=-1,O=0,k=0;k<T;k++)O<S[k]&&(O=S[k],E=k);for(X.FA(t.d,F[E]),c=0;c<T;c++)c!=E&&X.FA(t.d,F[c]);}}},X.vA=function(t){var A,e;for(A=0;A<t.f.q;A++)!t.b[A]&&n.Lo(t.f,A)==0&&(e=new X.uB(t.f,1,false),t.b[A]=true,e.e[0]=A,e.a[0]=0,e.b[0]=0,e.k[0]=0,X.FA(t.d,e));},X.wA=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S;for(b=null,b=new n.iu(),n.cu(b,t.f),D=B(Q.xfb,Tt,6,t.f.f,16,1),k=A.Tb();k.ed();)if(g=k.fd(),c=null,S=g.b,n._t(b,S),n.Pt(b,4,8)!=0&&(c=b.F),c)for(T=new s.trb(c);T.a<T.c.a.length;){for(y=s.srb(T),F=0,r=y,P=0,i=r.length;P<i;++P)e=r[P],D[e]&&++F;if(F<=1){for(o=g.c,t.a?o=false:t.a=true,l=new X.uB(t.f,y.length,o),U=0;U<y.length;U++)e=y[U],o&&n.sn(t.f,e,true),l.k[U]=256,l.e[U]=e,l.a[U]=n.Ql(g.b,U)/g.a,l.b[U]=n.Rl(g.b,U)/g.a,D[e]=true,t.b[e]=true;for(u=0;u<S.g;u++)t.c[n.Xo(t.f,y[S.D[0][u]],y[S.D[1][u]])]=true;X.FA(t.d,l);}}return D;},X.xA=function(t){var A,e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E;for(A=X.Rz(t),i=B(Q.O7,Y,6,t.f.r,15,1),X.sA(t,i,A),P=0;P<t.f.r;P++)i[P]==2&&(n.fq(t.f,n.Zl(t.f,0,P))||n.fq(t.f,n.Zl(t.f,1,P)))&&(i[P]=3);for(c=0;c<t.d.a.length;c++){for(y=s.xj(t.d,c),o=X.lB(y),F=y.c,k=new X.vB(y),g=-1,T=0;T<224&&o.a.length!=0;T++){if(l=s.swb(t.g,o.a.length),u=(w.HAb(l,o.a.length),o.a[l]),D=X.gA(t,u[0],u[1]),e=B(Q.S7,H,6,D.length,15,1),r=0,T<32)for(b=1;b<D.length-1;b++)i[D[b]]==3&&(e[r++]=D[b]);else if(T<96)for(b=1;b<D.length-1;b++)i[D[b]]>=2&&(e[r++]=D[b]);else for(b=1;b<D.length-1;b++)i[D[b]]>=1&&(e[r++]=D[b]);if(r!=0){if(R=e[0],r>1)do R=e[s.swb(t.g,r)];while(R==g);R!=g&&(g=R,X.iB(y,R),o=X.lB(y),F>y.c&&(F=y.c,k=new X.vB(y)));}}s.Hj(t.d,c,k),y=k,U=1;do{for(S=9999,b=0;b<y.e.length;b++)E=A[y.e[b]],E==U?X.rB(y,b):E>U&&E<S&&(S=E);U=S;}while(S!=9999);}},X.yA=function(t,A){t.g=new s.wwb(A);},X.zA=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R;for(P=B(Q.Q7,jt,6,n.Lo(t.f,e)+1,15,1),D=B(Q.S7,H,6,n.Lo(t.f,e)+1,15,1),u=B(Q.S7,H,6,n.Lo(t.f,e)+1,15,1),k=X.nB(A,e),i=0,l=0;l<n.Lo(t.f,e);l++)D[i]=n.Zo(t.f,e,l),u[i]=n._o(t.f,e,l),y=X.nB(A,D[i]),y!=-1&&(P[i++]=X.$A(A.a[k],A.b[k],A.a[y],A.b[y]));if(i==1)return P[0]+t1;for(U=i-1;U>0;U--)for(T=0;T<U;T++)P[T]>P[T+1]&&(F=P[T],P[T]=P[T+1],P[T+1]=F,S=D[T],D[T]=D[T+1],D[T+1]=S,R=u[T],u[T]=u[T+1],u[T+1]=R);for(P[i]=P[0]+N1,D[i]=D[0],u[i]=u[0],c=-100,b=0,o=0;o<i;o++)r=P[o+1]-P[o],i>2&&n.gq(t.f,u[o])&&n.gq(t.f,u[o+1])&&(g=X.iA(t,D[o],e,D[o+1]),g!=0&&(r-=100-g)),c<r&&(c=r,b=o);return (P[b]+P[b+1])/2;},X.AA=function(t,A,e,r){var P;P=d.Math.min(s.zj(t.d,A,0),s.zj(t.d,e,0)),s.rj(t.d,P,r),s.Fj(t.d,A),s.Fj(t.d,e);},X.BA=function(){X.CA.call(this,2);},X.CA=function(t){this.e=t,(t&1)==0&&!X.Oz&&!X.Oz&&(X.Oz=new X.dB());},X.DA=function(t,A){var e,r,P,i,D;for(D=0,e=0,r=0;r<A;r++)D+=t[r].b*d.Math.sin(t[r].a),e+=t[r].b*d.Math.cos(t[r].a);return e==0?i=D>0?F0:ue:(i=d.Math.atan(D/e),e<0&&(i+=t1)),P=d.Math.sqrt(D*D+e*e)/A,new X.YA(i,P);},X.EA=function(t,A){var e,r;return e=d.Math.abs(t.b),r=d.Math.abs(A.b),e<r?-1:e>r?1:0;},I(69,1,{},X.BA,X.CA),L.a=false,L.e=0,Q.i9=N(69),X.FA=function(t,A){var e,r;for(r=new s.trb(t);r.a<r.c.a.length;)if(e=s.srb(r),X.gB(e,A))return  false;return w.kAb(t.a,A),true;},X.GA=function(t,A){return X.FA(t,A);},X.HA=function(){s.Mj.call(this);},I(315,20,We,X.HA),L.add=function(t){return X.GA(this,t);},Q.e9=N(315),X.JA=function(t){var A,e;return A=t.c.e.length,e=t.c.e.length,e<A?-1:e>A?1:0;},X.KA=function(){},I(317,1,{},X.KA),L.Rb=function(t,A){var e;return X.JA((e=t,e));},L.mb=function(t){return this===t;},Q.f9=N(317),X.NA=function(t){var A,e;return A=t.c.e.length,e=t.c.e.length,A<e?-1:A>e?1:0;},X.OA=function(){},I(318,1,{},X.OA),L.Rb=function(t,A){var e;return X.NA((e=t,e));},L.mb=function(t){return this===t;},Q.g9=N(318),X.RA=function(){},I(316,1,{},X.RA),L.Rb=function(t,A){return X.EA(t,A);},L.mb=function(t){return this===t;},Q.h9=N(316),X.UA=function(t,A,e){t.c[0]+=X.jB(t.b[0],A),t.d[0]+=X.kB(t.b[0],A),t.c[1]+=X.jB(t.b[1],e),t.d[1]+=X.kB(t.b[1],e),++t.a[0],++t.a[1];},X.VA=function(t,A,e){var r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x;for(r=B(Q.Q7,jt,6,2,15,1),l=0;l<2;l++)t.c[l]/=t.a[l],t.d[l]/=t.a[l],r[l]=X.eB(t.b[l],t.c[l],t.d[l],t.b[1-l].e.length,A);for(X.sB(t.b[0],t.c[0],t.d[0],F0-r[0]),X.sB(t.b[1],t.c[1],t.d[1],4.71238898038469-r[1]),x=L0,O=oP,D=t.d[0]-t.d[1],U=0;U<t.b[1].b.length;U++)t.b[1].b[U]+=D,x>t.b[1].b[U]&&(x=t.b[1].b[U]),O<t.b[1].b[U]&&(O=t.b[1].b[U]);for(R=O-x+2*A,P=pt(d.Math.ceil(R)),x+=(R-P)/2-A,F=B(Q.Q7,jt,6,P,15,1),y=0;y<P;y++)F[y]=t.c[1]+A;for(T=0;T<t.b[1].b.length;T++)for(E=t.b[1].b[T]-x,S=pt(E-A),u=d.Math.min(pt(E+A),P-1),k=S;k<=u;k++)F[k]>t.b[1].a[T]&&(F[k]=t.b[1].a[T]);for(c=0;c<P;c++)F[c]-=A;for(i=t.c[0]-t.c[1],b=0;b<t.b[0].a.length;b++)g=pt(t.b[0].b[b]-x),g>=0&&g<F.length&&i<t.b[0].a[b]-F[g]&&(i=t.b[0].a[b]-F[g]);for(o=0;o<t.b[1].a.length;o++)t.b[1].a[o]+=i;e&&(X.sB(t.b[0],t.c[0],t.d[0],r[0]-F0),X.sB(t.b[1],t.c[0],t.d[0],r[0]-F0));},X.WA=function(t,A){var e,r;for(this.b=B(Q.o9,C1,47,2,0,1),this.b[0]=t,this.b[1]=A,this.c=B(Q.Q7,jt,6,2,15,1),this.d=B(Q.Q7,jt,6,2,15,1),this.a=B(Q.S7,H,6,2,15,1),e=0;e<2;e++){for(r=0;r<this.b[e].e.length;r++)this.c[e]+=X.jB(this.b[e],r),this.d[e]+=X.kB(this.b[e],r);this.a[e]=this.b[e].e.length;}},X.XA=function(t,A,e,r){this.b=B(Q.o9,C1,47,2,0,1),this.b[0]=t,this.b[1]=A,this.c=B(Q.Q7,jt,6,2,15,1),this.d=B(Q.Q7,jt,6,2,15,1),this.c[0]=X.jB(this.b[0],e),this.d[0]=X.kB(this.b[0],e),this.c[1]=X.jB(this.b[1],r),this.d[1]=X.kB(this.b[1],r),this.a=B(Q.S7,H,6,2,15,1),this.a[0]=1,this.a[1]=1;},I(106,1,{106:1},X.WA,X.XA),Q.j9=N(106),X.YA=function(t,A){this.a=t,this.b=A;},X.ZA=function(t,A,e,r){var P,i;this.a=X.$A(t,A,e,r),P=e-t,i=r-A,this.b=d.Math.sqrt(P*P+i*i);},X.$A=function(t,A,e,r){var P,i,D;return i=e-t,D=r-A,D!=0?(P=d.Math.atan(i/D),D<0&&(i<0?P-=t1:P+=t1)):P=i>0?F0:ue,P;},I(41,1,{41:1},X.YA,X.ZA),L.a=0,L.b=0,Q.k9=N(41),X._A=function(t){this.a=B(Q.S7,H,6,t,15,1),this.b=B(Q.S7,H,6,t,15,1);},I(247,1,{},X._A),Q.l9=N(247),X.aB=function(t,A,e){this.c=t,this.a=A,this.b=e;},I(197,1,{197:1},X.aB),L.a=0,L.b=0,Q.m9=N(197),X.cB=function(){X.cB=i1,X.bB=j(v(Q.Qcb,1),yt,2,6,["gkvt@@@@LddTTTrbTRTRTRRRRRRRRRRRRRrVRrIh\\IAaQxlY@gRHdJCJcRXlv_CfJx|A\\hRHejiLaQjTje^kSjtFcIhvXmVKMjt{lN{Kavy\\^wGjjjjjjjjjjjjjjjjjjjjjjjjjjjjjjh@@vo@HBC@PhLN@bPhtFKCcpDbILaRhtzCIbsX\\nOO`JDbqDjSKdJeJmQjtz}Ahr[LVkMnpz\\nwGj{PBhBdBlBBBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajijej` !BnkjyVwsVr|iQn|Q|goTZWPIJwbudnRkVYBez]siZymNJZUqNFBqZWxS~iCXVU]SeRjwrtSPAjkvXLpBAZauDPzq]PfMlecrMnkv|@\\SFD`m|mWiEoCXp`SIe_J[l|[XCbloTV`[Gc@FJGopyyoOlFQfUy^w\\Bgz|","gcrt@@@@LdbbbbTRbRbRbRRRRRRRRRRRRVRrVQIA`HtRGAaIxZAHfShTjCIbqylQGKgqdBaXeQJeruBiPitZmFoPZLFSYbvZlVGMnsZ]vWSmr{]UUUUUUUUUUUUUUUUUUUUUUUUUUUUUT@@[G`DAA`HTFG@QHTZCEaqxBQDfPiTZ]AdqYlNWGgpEBQXbUIerEReVhuZ]^`tYMfKUfwX]NW[jkPBhBdBlBbBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajij` !B^cR]`]Fm]QkfljE\\p\x7FUVfgOmFXsQe_gXPyXis_wF|vUUX_XbxpzU]HUFgYViwFo~@uemc@}~T\x7FIEPioYVwr]JnM~[ZEC\\g}~o_pUfdo~irsklTLiyVJshnw^iVAsZ`_~}PYkckURH{FYMImFaRaccUlCZSHMfP","dml@@Dje^VGiyZjjjh@vtHSBinFU@ !BPTCTy[skMzUPF`AJbBixEZHS[Il","dml@@DjYVvGiyZjjjh@vtHSBinFU@ !BwLo~BJ~UquhXBinZ\\ykA@F_eMrT","dml@@LdfbbQX^fUZjjj`C[PaLJfxYT !BzxIHVc{OiJVRpprePho~]}y\x7FwLl","deL@@DjUYkfEijjjj@MeBDpj[ad !B\x7FaA\x7FMVr[AvkKzm_jKvVbD{sk","dil@@LddTQRl[NX^Fjjjj@MiBDpj[a@ !BPfL@\x7Fox@M~T@\x7Fox@\x7F`C~@@","daL@@DjYtKJqjynjjjj@MaBDpj[` !B`bL@_gx@@Gy~@Gx@_`@"]);},X.dB=function(){X.cB();var t,A,e,r,P,i,D;for(s.Mj.call(this),i=new n.yu(),e=X.bB,r=0,P=e.length;r<P;++r)A=e[r],t=n.Pq(new n.Xq(),A),n.ru(i,t),D=new X.wB(t),D.a=n.Ul(D.b),w.kAb(this.a,D);},I(369,20,We,X.dB),Q.n9=N(369),X.eB=function(t,A,e,r,P){var i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot,K;if(t.e.length==1)return 0;for(V=P+d.Math.sqrt(r),U=B(Q.Q7,jt,6,36,15,1),b=0;b<t.e.length;b++)i=X.$A(A,e,t.a[b],t.b[b]),u=X.fB(zt(w1(d.Math.round(i*36/N1)))),y=A-t.a[b],T=e-t.b[b],ot=y*y+T*T,U[u]<ot&&(U[u]=ot);for(O=-1,E=-1,g=0;g<36;g++)U[g]=d.Math.sqrt(U[g]),O<U[g]&&(O=U[g],E=g);for(At=E-18<0?E-18+36:E-18>=36?E-18-36:E-18,k=0;k<=18;k++)U[At+k<0?At+k+36:At+k>=36?At+k-36:At+k]+=.01*k,U[At-k<0?At-k+36:At-k>=36?At-k-36:At-k]+=.01*k;for(K=B(Q.Q7,jt,6,9,15,1),o=B(Q.Q7,jt,6,9,15,1),F=1;F<9;F++)K[F]=d.Math.sin(F*r9),o[F]=d.Math.cos(F*r9);for(G=L0,x=-1,D=0;D<36;D++)if(!(U[D]>=G)){for(R=U[D],c=1;c<9;c++){for(S=-1;S<=1&&(J=D+S*c<0?D+S*c+36:D+S*c>=36?D+S*c-36:D+S*c,!(!(U[J]*o[c]<=R)&&(l=o[c]*d.Math.min(U[J],V/K[c]),R<l&&(R=l,G<=R))));S+=2);if(G<=R)break;}G>R&&(G=R,x=D);}return N1*x/36;},X.fB=function(t){return t<0?t+36:t>=36?t-36:t;},X.gB=function(t,A){var e,r,P;if(A.e.length!=t.e.length)return  false;for(r=X.oB(t),P=X.oB(A),e=0;e<r.length;e++)if(r[e]!=P[e])return  false;return  true;},X.hB=function(t,A,e){var r,P,i;for(i=0;i<t.e.length;i++)P=d.Math.sqrt((t.a[i]-A)*(t.a[i]-A)+(t.b[i]-e)*(t.b[i]-e)),r=0-X.$A(A,e,t.a[i],t.b[i]),t.a[i]=A+P*d.Math.sin(r),t.b[i]=e+P*d.Math.cos(r);},X.iB=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x;if(t.d==null&&(t.d=B(Q.S7,E1,7,t.j.r,0,2)),t.d[A]==null){for(c=B(Q.S7,H,6,t.e.length,15,1),R=B(Q.xfb,Tt,6,t.j.q,16,1),P=n.Zl(t.j,0,A),i=n.Zl(t.j,1,A),c[0]=P,R[P]=true,U=0,b=0;U<=b;){for(k=0;k<n.Lo(t.j,c[U]);k++)D=n.Zo(t.j,c[U],k),!R[D]&&D!=i&&(c[++b]=D,R[D]=true);if(U==b)break;++U;}if(T=b+1>(t.e.length/2|0),t.i){for(o=false,u=false,k=0;k<t.e.length;k++)r=t.e[k],n.Km(t.j,r)&&r!=P&&r!=i&&(R[t.e[k]]?o=true:u=true);o!=u&&(T=o);}for(l=2,t.d[A]=B(Q.S7,H,6,T?t.e.length-b:b+2,15,1),F=0;F<t.e.length;F++)t.e[F]==P?t.d[A][T?0:1]=F:t.e[F]==i?t.d[A][T?1:0]=F:T^R[t.e[F]]&&(t.d[A][l++]=F);}for(O=t.a[t.d[A][0]],x=t.b[t.d[A][0]],E=X.$A(O,x,t.a[t.d[A][1]],t.b[t.d[A][1]]),g=2;g<t.d[A].length;g++)S=t.d[A][g],y=d.Math.sqrt((t.a[S]-O)*(t.a[S]-O)+(t.b[S]-x)*(t.b[S]-x)),e=2*E-X.$A(O,x,t.a[S],t.b[S]),t.a[S]=O+y*d.Math.sin(e),t.b[S]=x+y*d.Math.cos(e);},X.jB=function(t,A){return t.a[A];},X.kB=function(t,A){return t.b[A];},X.lB=function(t){var A,e,r,P,i,D,u,o;for(t.c=0,e=new s.Mj(),P=1;P<t.e.length;P++)for(i=0;i<P;i++)u=d.Math.abs(t.a[P]-t.a[i]),o=d.Math.abs(t.b[P]-t.b[i]),r=d.Math.sqrt(u*u+o*o),r<.8&&(A=B(Q.S7,H,6,2,15,1),A[0]=t.e[P],A[1]=t.e[i],w.kAb(e.a,A)),D=1-d.Math.min(r,1),t.c+=D*D;return e;},X.mB=function(t,A){return t.e[A];},X.nB=function(t,A){var e;for(e=0;e<t.e.length;e++)if(A==t.e[e])return e;return  -1;},X.oB=function(t){return t.n==null&&(t.n=s.Brb(t.e,t.e.length),w.nAb(t.n,w.UAb(T0(s.nsb.prototype.ld,s.nsb,[])))),t.n;},X.pB=function(t,A){var e;for(e=0;e<t.e.length;e++)if(A==t.e[e])return  true;return  false;},X.qB=function(t){var A,e,r,P,i,D;for(r=0,i=0;i<t.e.length;i++)for(A=t.e[i],e=n.Lo(t.j,A),D=0;D<e;D++)n.Zo(t.j,A,D)>A&&++r;for(t.f=B(Q.S7,H,6,r,15,1),t.g=B(Q.S7,H,6,t.j.q,15,1),r=0,P=0;P<t.e.length;P++)for(A=t.e[P],e=n.Lo(t.j,A),t.g[A]=P,D=0;D<e;D++)n.Zo(t.j,A,D)>A&&(t.f[r++]=n._o(t.j,A,D));},X.rB=function(t,A){var e,r,P,i,D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O;for(b=t.a[A],S=t.b[A],i=B(Q.k9,C1,41,4,0,1),U=0,y=0;y<t.f.length&&!(U>=4);y++)if(!(A==t.g[n.Zl(t.j,0,t.f[y])]||A==t.g[n.Zl(t.j,1,t.f[y])])){if(g=t.a[t.g[n.Zl(t.j,0,t.f[y])]],R=t.b[t.g[n.Zl(t.j,0,t.f[y])]],k=t.a[t.g[n.Zl(t.j,1,t.f[y])]],E=t.b[t.g[n.Zl(t.j,1,t.f[y])]],u=d.Math.sqrt((g-b)*(g-b)+(R-S)*(R-S)),o=d.Math.sqrt((k-b)*(k-b)+(E-S)*(E-S)),P=d.Math.sqrt((k-g)*(k-g)+(E-R)*(E-R)),u<P&&o<P){g==k?(D=d.Math.abs(b-g),D<.5&&(i[U++]=new X.YA(X.$A(g,S,b,S),(.5-D)/2))):R==E?(D=d.Math.abs(S-R),D<.5&&(i[U++]=new X.YA(X.$A(b,R,b,S),(.5-D)/2))):(T=(E-R)/(k-g),c=-1/T,e=R-T*g,r=S-c*b,F=(r-e)/(T-c),O=T*F+e,D=d.Math.sqrt((F-b)*(F-b)+(O-S)*(O-S)),D<.5&&(i[U++]=new X.YA(X.$A(F,O,b,S),(.5-D)/2)));continue;}if(u<.5){i[U++]=new X.YA(X.$A(g,R,b,S),(.5-u)/2);continue;}if(o<.5){i[U++]=new X.YA(X.$A(k,E,b,S),(.5-o)/2);continue;}}U>0&&(l=X.DA(i,U),t.a[A]+=l.b*d.Math.sin(l.a),t.b[A]+=l.b*d.Math.cos(l.a));},X.sB=function(t,A,e,r){var P,i,D;for(D=0;D<t.e.length;D++)i=d.Math.sqrt((t.a[D]-A)*(t.a[D]-A)+(t.b[D]-e)*(t.b[D]-e)),P=X.$A(A,e,t.a[D],t.b[D])+r,t.a[D]=A+i*d.Math.sin(P),t.b[D]=e+i*d.Math.cos(P);},X.tB=function(t,A,e){var r;for(r=0;r<t.e.length;r++)t.a[r]+=A,t.b[r]+=e;},X.uB=function(t,A,e){this.j=t,this.i=e,this.e=B(Q.S7,H,6,A,15,1),this.k=B(Q.S7,H,6,A,15,1),this.a=B(Q.Q7,jt,6,A,15,1),this.b=B(Q.Q7,jt,6,A,15,1);},X.vB=function(t){var A,e;for(this.j=t.j,this.i=t.i,this.e=B(Q.S7,H,6,t.e.length,15,1),this.k=B(Q.S7,H,6,t.e.length,15,1),this.a=B(Q.Q7,jt,6,t.e.length,15,1),this.b=B(Q.Q7,jt,6,t.e.length,15,1),e=0;e<t.e.length;e++)this.e[e]=t.e[e],this.k[e]=t.k[e],this.a[e]=t.a[e],this.b[e]=t.b[e];if(t.f!=null)for(this.f=B(Q.S7,H,6,t.f.length,15,1),A=0;A<t.f.length;A++)this.f[A]=t.f[A];if(t.g!=null)for(this.g=B(Q.S7,H,6,t.g.length,15,1),A=0;A<t.g.length;A++)this.g[A]=t.g[A];},I(47,1,{47:1},X.uB,X.vB),L.c=0,L.i=false,Q.o9=N(47),X.wB=function(t){this.b=t,this.c=false;},I(246,1,{246:1},X.wB),L.a=0,L.c=false,Q.p9=N(246),gA.xB=function(t){t.c=new s.Mj();},gA.yB=function(t,A,e,r,P,i){var D,u,o,l,U,y,T,c,b,g,k,F,S,R,E,O,x,G,J,V,At,ot;for(R=-1,G=0,x=0,V=0,k=0,F=0,S=0,At=0,ot=0,G=0,U=0;U<t.d;U++)G+=r[U]*r[U];if(G=d.Math.sqrt(G),G>i)for(y=0;y<t.d;y++)r[y]*=i/G;for(x=0,T=0;T<t.d;T++)x+=r[T]*t.e[T];if(x>=0)return R;for(V=0,c=0;c<t.d;c++)J=d.Math.abs(r[c])/d.Math.max(d.Math.abs(A[c]),1),J>V&&(V=J);for(S=1e-7/V,k=1,g=0;g<1e3;){if(k<S){R=1;break;}for(b=0;b<t.d;b++)P[b]=A[b]+k*r[b];if(t.k=Z.xC(t,P),t.k-e<=we*k*x)return 0;g==0?At=-x/(2*(t.k-e-x)):(E=t.k-e-k*x,O=ot-e-F*x,D=(E/(k*k)-O/(F*F))/(k-F),u=(-F*E/(k*k)+k*O/(F*F))/(k-F),D==0?At=-x/(2*u):(o=u*u-3*D*x,o<0?At=.5*k:u<=0?At=(-u+d.Math.sqrt(o))/(3*D):At=-x/(u+d.Math.sqrt(o))),At>.5*k&&(At=.5*k)),F=k,ot=t.k,k=d.Math.max(At,.1*k),++g;}for(l=0;l<t.d;l++)P[l]=A[l];return R;},gA.zB=function(t,A,e){var r,P,i,D;for(D=1,P=0;P<t.g.q;P++)t.i[3*P]=