/*
 * openchemlib-utils v8.7.0
 * Various utilities that extends openchemlib-js like HOSE codes or diastereotopic IDs
 * https://github.com/cheminfo/openchemlib-utils#readme
 *
 * Licensed under the MIT license.
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.OCLUtils = {}));
})(this, (function (exports) { 'use strict';

    var _documentCurrentScript = typeof document !== 'undefined' ? document.currentScript : null;
    let xAtomicNumber = 0;
    /**
     * Returns the atomic number of the X atom
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @returns
     */
    function getXAtomicNumber(molecule) {
      if (!xAtomicNumber) {
        const OCL = molecule.getOCL();
        xAtomicNumber = OCL.Molecule.getAtomicNoFromLabel('X', OCL.Molecule.cPseudoAtomX);
      }
      return xAtomicNumber;
    }

    /**
     * Check if a specific atom is a sp3 carbon
     * @param {import('openchemlib').Molecule} molecule
     * @param {number} atomID
     */
    function isCsp3(molecule, atomID) {
      if (molecule.getAtomicNo(atomID) !== 6) return false;
      if (molecule.getAtomCharge(atomID) !== 0) return false;
      if (molecule.getImplicitHydrogens(atomID) + molecule.getConnAtoms(atomID) !== 4) {
        return false;
      }
      return true;
    }

    /**
     * This method put all the chiral centers in the molecule in the same group and racemic
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @param {object} [options={}]
     * @param {object} [options.OCL] - openchemlib library
     */
    function makeRacemic(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      // if we don't calculate this we have 2 epimers
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
      // we need to make one group "AND" for chiral (to force to racemic, this means diastereotopic and not enantiotopic)
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (molecule.getAtomParity(i) !== Molecule.cAtomParityNone) {
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0); // changed to group 0; TLS 9.Nov.2015
        }
      }
      // after the change we need to recalculate the CIP
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
    }

    /**
     * Tag an atom to be able to visualize it
     * @param molecule
     * @param iAtom
     */
    function tagAtom(molecule, iAtom) {
      const customLabel = `${molecule.getAtomLabel(iAtom)}*`;
      molecule.setAtomCustomLabel(iAtom, customLabel);
      if (molecule.getAtomicNo(iAtom) === 1) {
        molecule.setAtomicNo(iAtom, getXAtomicNumber(molecule));
      } else {
        // we can not use X because we would have problems with valencies if it is
        // expanded hydrogens or not
        // we can not only use a custom label because it does not count for the canonisation
        molecule.setAtomMass(iAtom, molecule.getAtomMass(iAtom) + 5);
      }
      return customLabel;
    }

    const FULL_HOSE_CODE$1 = 1;
    const HOSE_CODE_CUT_C_SP3_SP3$1 = 2;
    /**
     * Returns an array of hose code fragments for the specified molecule.
     * @param molecule - The OCL molecule to process.
     * @param options - Options for generating hose codes.
     * @returns An array of hose code fragments.
     */
    function getHoseCodesForAtomsAsFragments(molecule, options = {}) {
      const OCL = molecule.getOCL();
      const {
        allowedCustomLabels,
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE$1,
        tagAtoms = [],
        tagAtomFct = tagAtom
      } = options;
      const rootAtoms = options.rootAtoms ? options.rootAtoms.slice() : [];
      molecule = molecule.getCompactCopy();
      if (tagAtoms.length > 0) {
        internalTagAtoms(molecule, tagAtoms, rootAtoms, tagAtomFct);
      } else {
        // this force reordering of atoms in order to have hydrogens at the end
        molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      }
      if (rootAtoms.length === 0) {
        for (let j = 0; j < molecule.getAllAtoms(); j++) {
          if (allowedCustomLabels?.includes(molecule.getAtomCustomLabel(j) ?? '') || molecule.getAtomCustomLabel(j)) {
            rootAtoms.push(j);
          }
        }
      }
      const fragments = [];
      // keep track of the atoms when creating the fragment
      const mappings = [];
      let min = 0;
      let max = 0;
      const atomMask = new Array(molecule.getAllAtoms()).fill(false);
      const atomList = new Array(molecule.getAllAtoms());
      for (let sphere = 0; sphere <= maxSphereSize; sphere++) {
        if (max === 0) {
          for (const rootAtom of rootAtoms) {
            atomList[max] = rootAtom;
            atomMask[rootAtom] = true;
            max++;
          }
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            const atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              const connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                switch (kind) {
                  case FULL_HOSE_CODE$1:
                    atomMask[connAtom] = true;
                    atomList[newMax++] = connAtom;
                    break;
                  case HOSE_CODE_CUT_C_SP3_SP3$1:
                    if (!(isCsp3(molecule, atom) && isCsp3(molecule, connAtom))) {
                      atomMask[connAtom] = true;
                      atomList[newMax++] = connAtom;
                    }
                    break;
                  default:
                    throw new Error('getHoseCoesForAtom unknown kind');
                }
              }
            }
          }
          min = max;
          max = newMax;
        }
        if (sphere >= minSphereSize) {
          const fragment = new OCL.Molecule(0, 0);
          molecule.copyMoleculeByAtoms(fragment, atomMask, true, mappings);
          // we using atomMapNo field in order to keep track of the original atom number even if we remove hydrogens
          for (let i = 0; i < fragment.getAllAtoms(); i++) {
            fragment.setAtomMapNo(i, mappings.indexOf(i) + 1);
          }
          fragment.removeExplicitHydrogens();
          makeRacemic(fragment);
          // we encode atom characteristics in the query features
          addQueryFeaturesAndRemoveMapNo(fragment, molecule);
          fragments.push(fragment);
        }
      }
      return fragments;
    }
    /**
     * If the atom is not an halogen, X or an hydrogen
     * we add query features to the atom
     * This includes aromaticity, ring size, number of hydrogens
     * @param fragment
     * @param molecule
     */
    function addQueryFeaturesAndRemoveMapNo(fragment, molecule) {
      const Molecule = molecule.getOCL().Molecule;
      for (let i = 0; i < fragment.getAllAtoms(); i++) {
        const mapping = fragment.getAtomMapNo(i) - 1;
        fragment.setAtomMapNo(i, 0);
        if ([1, 9, 17, 35, 53, getXAtomicNumber(molecule)].includes(fragment.getAtomicNo(i))) {
          continue;
        }
        // aromaticity
        const isAromatic = molecule.isAromaticAtom(mapping);
        if (isAromatic) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFAromatic, true);
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNotAromatic, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFAromatic, false);
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNotAromatic, true);
        }
        // cycles
        const smallestRing = molecule.getAtomRingSize(mapping);
        switch (smallestRing) {
          case 0:
            break;
          case 3:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize3, true);
            break;
          case 4:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize4, true);
            break;
          case 5:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize5, true);
            break;
          case 6:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize6, true);
            break;
          case 7:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize7, true);
            break;
          default:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSizeLarge, true);
        }
        const nbHydrogens = molecule.getAllHydrogens(mapping);
        if (nbHydrogens === 0) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot0Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot0Hydrogen, true);
        }
        if (nbHydrogens === 1) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot1Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot1Hydrogen, true);
        }
        if (nbHydrogens === 2) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot2Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot2Hydrogen, true);
        }
        if (nbHydrogens === 3) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot3Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot3Hydrogen, true);
        }
      }
    }
    // tagging atoms may change the order of the atoms because hydrogens must be at the end of the file
    // in order to remember the rootAtoms we will tag before
    function internalTagAtoms(molecule, tagAtoms, rootAtoms, tagAtomFct) {
      const OCL = molecule.getOCL();
      if (tagAtoms) {
        for (let i = 0; i < molecule.getAllAtoms(); i++) {
          molecule.setAtomMapNo(i, i + 1);
        }
        if (tagAtoms.length > 0) {
          for (const atom of tagAtoms) {
            tagAtomFct(molecule, atom);
          }
        }
      }
      // this force reordering of atoms in order to have hydrogens at the end
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      if (rootAtoms.length > 0) {
        const mapping = new Int32Array(molecule.getAllAtoms());
        for (let i = 0; i < molecule.getAllAtoms(); i++) {
          mapping[molecule.getAtomMapNo(i) - 1] = i;
        }
        for (let i = 0; i < rootAtoms.length; i++) {
          rootAtoms[i] = mapping[rootAtoms[i]];
        }
      }
    }

    /**
     * We need to create an array of atoms
     * that contains an array of pathLength
     * that contains an array of object
     * @param molecule
     * @param options
     * @returns
     */
    function getAllAtomsPaths(molecule, options = {}) {
      const {
        maxPathLength = 5
      } = options;
      const allAtomsPaths = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        const oneAtomPaths = [];
        allAtomsPaths.push(oneAtomPaths);
        let atomPaths = [];
        atomPaths.push({
          path: [i],
          pathLength: 0
        });
        oneAtomPaths.push(atomPaths);
        let nextIndexes = [0];
        let nextAtoms = [i];
        for (let sphere = 1; sphere <= maxPathLength; sphere++) {
          atomPaths = [];
          oneAtomPaths.push(atomPaths);
          const currentIndexes = nextIndexes;
          const currentAtoms = nextAtoms;
          nextIndexes = [];
          nextAtoms = [];
          for (let i = 0; i < currentIndexes.length; i++) {
            const atom = currentAtoms[i];
            const index = currentIndexes[i];
            const previousPath = oneAtomPaths[sphere - 1]?.[index]?.path;
            if (!previousPath) {
              throw new Error(`Unexpected missing previousPath for sphere ${sphere - 1} and index ${index}`);
            }
            for (let conn = 0; conn < molecule.getAllConnAtoms(atom); conn++) {
              const connectedAtom = molecule.getConnAtom(atom, conn);
              if (previousPath.includes(connectedAtom)) continue;
              nextIndexes.push(atomPaths.length);
              nextAtoms.push(connectedAtom);
              atomPaths.push({
                path: [...previousPath, connectedAtom],
                pathLength: sphere
              });
            }
          }
        }
      }
      return allAtomsPaths;
    }

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    function getAugmentedNamespace(n) {
      if (Object.prototype.hasOwnProperty.call(n, '__esModule')) return n;
      var f = n.default;
    	if (typeof f == "function") {
    		var a = function a () {
    			var isInstance = false;
          try {
            isInstance = this instanceof a;
          } catch {}
    			if (isInstance) {
            return Reflect.construct(f, arguments, this.constructor);
    			}
    			return f.apply(this, arguments);
    		};
    		a.prototype = f.prototype;
      } else a = {};
      Object.defineProperty(a, '__esModule', {value: true});
    	Object.keys(n).forEach(function (k) {
    		var d = Object.getOwnPropertyDescriptor(n, k);
    		Object.defineProperty(a, k, d.get ? d : {
    			enumerable: true,
    			get: function () {
    				return n[k];
    			}
    		});
    	});
    	return a;
    }

    var matrix = {};

    // eslint-disable-next-line @typescript-eslint/unbound-method
    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
     *
     * @param value - Object to check.
     * @returns True if the object is an array or a typed array.
     */
    function isAnyArray$1(value) {
      const tag = toString.call(value);
      return tag.endsWith('Array]') && !tag.includes('Big');
    }

    var libEsm = /*#__PURE__*/Object.freeze({
        __proto__: null,
        isAnyArray: isAnyArray$1
    });

    var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

    function max(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var maxValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] > maxValue) maxValue = input[i];
      }
      return maxValue;
    }

    function min(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var minValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] < minValue) minValue = input[i];
      }
      return minValue;
    }

    function rescale$1(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      } else if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var output;
      if (options.output !== undefined) {
        if (!isAnyArray$1(options.output)) {
          throw new TypeError('output option must be an array if specified');
        }
        output = options.output;
      } else {
        output = new Array(input.length);
      }
      var currentMin = min(input);
      var currentMax = max(input);
      if (currentMin === currentMax) {
        throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
      }
      var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
      if (minValue >= maxValue) {
        throw new RangeError('min option must be smaller than max option');
      }
      var factor = (maxValue - minValue) / (currentMax - currentMin);
      for (var i = 0; i < input.length; i++) {
        output[i] = (input[i] - currentMin) * factor + minValue;
      }
      return output;
    }

    var libEs6 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        default: rescale$1
    });

    var require$$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

    Object.defineProperty(matrix, '__esModule', {
      value: true
    });
    var isAnyArray = require$$0;
    var rescale = require$$1;
    const indent = ' '.repeat(2);
    const indentData = ' '.repeat(4);

    /**
     * @this {Matrix}
     * @returns {string}
     */
    function inspectMatrix() {
      return inspectMatrixWithOptions(this);
    }
    function inspectMatrixWithOptions(matrix, options = {}) {
      const {
        maxRows = 15,
        maxColumns = 10,
        maxNumSize = 8,
        padMinus = 'auto'
      } = options;
      return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
    }
    function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
      const {
        rows,
        columns
      } = matrix;
      const maxI = Math.min(rows, maxRows);
      const maxJ = Math.min(columns, maxColumns);
      const result = [];
      if (padMinus === 'auto') {
        padMinus = false;
        loop: for (let i = 0; i < maxI; i++) {
          for (let j = 0; j < maxJ; j++) {
            if (matrix.get(i, j) < 0) {
              padMinus = true;
              break loop;
            }
          }
        }
      }
      for (let i = 0; i < maxI; i++) {
        let line = [];
        for (let j = 0; j < maxJ; j++) {
          line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
        }
        result.push(`${line.join(' ')}`);
      }
      if (maxJ !== columns) {
        result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
      }
      if (maxI !== rows) {
        result.push(`... ${rows - maxRows} more rows`);
      }
      return result.join(`\n${indentData}`);
    }
    function formatNumber(num, maxNumSize, padMinus) {
      return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
    }
    function formatNumber2(num, len) {
      // small.length numbers should be as is
      let str = num.toString();
      if (str.length <= len) return str;

      // (7)'0.00123' is better then (7)'1.23e-2'
      // (8)'0.000123' is worse then (7)'1.23e-3',
      let fix = num.toFixed(len);
      if (fix.length > len) {
        fix = num.toFixed(Math.max(0, len - (fix.length - len)));
      }
      if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
        return fix;
      }

      // well, if it's still too long the user should've used longer numbers
      let exp = num.toExponential(len);
      if (exp.length > len) {
        exp = num.toExponential(Math.max(0, len - (exp.length - len)));
      }
      return exp.slice(0);
    }
    function installMathOperations(AbstractMatrix, Matrix) {
      AbstractMatrix.prototype.add = function add(value) {
        if (typeof value === 'number') return this.addS(value);
        return this.addM(value);
      };
      AbstractMatrix.prototype.addS = function addS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.addM = function addM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.add = function add(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.add(value);
      };
      AbstractMatrix.prototype.sub = function sub(value) {
        if (typeof value === 'number') return this.subS(value);
        return this.subM(value);
      };
      AbstractMatrix.prototype.subS = function subS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.subM = function subM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.sub = function sub(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sub(value);
      };
      AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
      AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
      AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
      AbstractMatrix.subtract = AbstractMatrix.sub;
      AbstractMatrix.prototype.mul = function mul(value) {
        if (typeof value === 'number') return this.mulS(value);
        return this.mulM(value);
      };
      AbstractMatrix.prototype.mulS = function mulS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.mulM = function mulM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mul = function mul(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mul(value);
      };
      AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
      AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
      AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
      AbstractMatrix.multiply = AbstractMatrix.mul;
      AbstractMatrix.prototype.div = function div(value) {
        if (typeof value === 'number') return this.divS(value);
        return this.divM(value);
      };
      AbstractMatrix.prototype.divS = function divS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.divM = function divM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.div = function div(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.div(value);
      };
      AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
      AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
      AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
      AbstractMatrix.divide = AbstractMatrix.div;
      AbstractMatrix.prototype.mod = function mod(value) {
        if (typeof value === 'number') return this.modS(value);
        return this.modM(value);
      };
      AbstractMatrix.prototype.modS = function modS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.modM = function modM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mod = function mod(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mod(value);
      };
      AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
      AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
      AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
      AbstractMatrix.modulus = AbstractMatrix.mod;
      AbstractMatrix.prototype.and = function and(value) {
        if (typeof value === 'number') return this.andS(value);
        return this.andM(value);
      };
      AbstractMatrix.prototype.andS = function andS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.andM = function andM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.and = function and(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.and(value);
      };
      AbstractMatrix.prototype.or = function or(value) {
        if (typeof value === 'number') return this.orS(value);
        return this.orM(value);
      };
      AbstractMatrix.prototype.orS = function orS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.orM = function orM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.or = function or(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.or(value);
      };
      AbstractMatrix.prototype.xor = function xor(value) {
        if (typeof value === 'number') return this.xorS(value);
        return this.xorM(value);
      };
      AbstractMatrix.prototype.xorS = function xorS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.xorM = function xorM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.xor = function xor(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.xor(value);
      };
      AbstractMatrix.prototype.leftShift = function leftShift(value) {
        if (typeof value === 'number') return this.leftShiftS(value);
        return this.leftShiftM(value);
      };
      AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.leftShift = function leftShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.leftShift(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
        if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
        return this.signPropagatingRightShiftM(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.signPropagatingRightShift(value);
      };
      AbstractMatrix.prototype.rightShift = function rightShift(value) {
        if (typeof value === 'number') return this.rightShiftS(value);
        return this.rightShiftM(value);
      };
      AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.rightShift = function rightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.rightShift(value);
      };
      AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
      AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
      AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
      AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
      AbstractMatrix.prototype.not = function not() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, ~this.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.not = function not(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.not();
      };
      AbstractMatrix.prototype.abs = function abs() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.abs(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.abs = function abs(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.abs();
      };
      AbstractMatrix.prototype.acos = function acos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acos = function acos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acos();
      };
      AbstractMatrix.prototype.acosh = function acosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acosh = function acosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acosh();
      };
      AbstractMatrix.prototype.asin = function asin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asin = function asin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asin();
      };
      AbstractMatrix.prototype.asinh = function asinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asinh = function asinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asinh();
      };
      AbstractMatrix.prototype.atan = function atan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atan = function atan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atan();
      };
      AbstractMatrix.prototype.atanh = function atanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atanh = function atanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atanh();
      };
      AbstractMatrix.prototype.cbrt = function cbrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cbrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cbrt = function cbrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cbrt();
      };
      AbstractMatrix.prototype.ceil = function ceil() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.ceil(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.ceil = function ceil(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.ceil();
      };
      AbstractMatrix.prototype.clz32 = function clz32() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.clz32(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.clz32 = function clz32(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.clz32();
      };
      AbstractMatrix.prototype.cos = function cos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cos = function cos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cos();
      };
      AbstractMatrix.prototype.cosh = function cosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cosh = function cosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cosh();
      };
      AbstractMatrix.prototype.exp = function exp() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.exp(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.exp = function exp(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.exp();
      };
      AbstractMatrix.prototype.expm1 = function expm1() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.expm1(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.expm1 = function expm1(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.expm1();
      };
      AbstractMatrix.prototype.floor = function floor() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.floor(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.floor = function floor(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.floor();
      };
      AbstractMatrix.prototype.fround = function fround() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.fround(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.fround = function fround(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.fround();
      };
      AbstractMatrix.prototype.log = function log() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log = function log(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log();
      };
      AbstractMatrix.prototype.log1p = function log1p() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log1p(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log1p = function log1p(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log1p();
      };
      AbstractMatrix.prototype.log10 = function log10() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log10(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log10 = function log10(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log10();
      };
      AbstractMatrix.prototype.log2 = function log2() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log2(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log2 = function log2(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log2();
      };
      AbstractMatrix.prototype.round = function round() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.round(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.round = function round(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.round();
      };
      AbstractMatrix.prototype.sign = function sign() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sign(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sign = function sign(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sign();
      };
      AbstractMatrix.prototype.sin = function sin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sin = function sin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sin();
      };
      AbstractMatrix.prototype.sinh = function sinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sinh = function sinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sinh();
      };
      AbstractMatrix.prototype.sqrt = function sqrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sqrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sqrt = function sqrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sqrt();
      };
      AbstractMatrix.prototype.tan = function tan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tan = function tan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tan();
      };
      AbstractMatrix.prototype.tanh = function tanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tanh = function tanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tanh();
      };
      AbstractMatrix.prototype.trunc = function trunc() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.trunc(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.trunc = function trunc(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.trunc();
      };
      AbstractMatrix.pow = function pow(matrix, arg0) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.pow(arg0);
      };
      AbstractMatrix.prototype.pow = function pow(value) {
        if (typeof value === 'number') return this.powS(value);
        return this.powM(value);
      };
      AbstractMatrix.prototype.powS = function powS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.powM = function powM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** matrix.get(i, j));
          }
        }
        return this;
      };
    }

    /**
     * @private
     * Check that a row index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkRowIndex(matrix, index, outer) {
      let max = outer ? matrix.rows : matrix.rows - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Row index out of range');
      }
    }

    /**
     * @private
     * Check that a column index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkColumnIndex(matrix, index, outer) {
      let max = outer ? matrix.columns : matrix.columns - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Column index out of range');
      }
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkRowVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.columns) {
        throw new RangeError('vector size must be the same as the number of columns');
      }
      return vector;
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkColumnVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.rows) {
        throw new RangeError('vector size must be the same as the number of rows');
      }
      return vector;
    }
    function checkRowIndices(matrix, rowIndices) {
      if (!isAnyArray.isAnyArray(rowIndices)) {
        throw new TypeError('row indices must be an array');
      }
      for (let i = 0; i < rowIndices.length; i++) {
        if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
          throw new RangeError('row indices are out of range');
        }
      }
    }
    function checkColumnIndices(matrix, columnIndices) {
      if (!isAnyArray.isAnyArray(columnIndices)) {
        throw new TypeError('column indices must be an array');
      }
      for (let i = 0; i < columnIndices.length; i++) {
        if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
          throw new RangeError('column indices are out of range');
        }
      }
    }
    function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
      if (arguments.length !== 5) {
        throw new RangeError('expected 4 arguments');
      }
      checkNumber('startRow', startRow);
      checkNumber('endRow', endRow);
      checkNumber('startColumn', startColumn);
      checkNumber('endColumn', endColumn);
      if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
        throw new RangeError('Submatrix indices are out of range');
      }
    }
    function newArray(length, value = 0) {
      let array = [];
      for (let i = 0; i < length; i++) {
        array.push(value);
      }
      return array;
    }
    function checkNumber(name, value) {
      if (typeof value !== 'number') {
        throw new TypeError(`${name} must be a number`);
      }
    }
    function checkNonEmpty(matrix) {
      if (matrix.isEmpty()) {
        throw new Error('Empty matrix has no elements to index');
      }
    }
    function sumByRow(matrix) {
      let sum = newArray(matrix.rows);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumByColumn(matrix) {
      let sum = newArray(matrix.columns);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumAll(matrix) {
      let v = 0;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v += matrix.get(i, j);
        }
      }
      return v;
    }
    function productByRow(matrix) {
      let sum = newArray(matrix.rows, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productByColumn(matrix) {
      let sum = newArray(matrix.columns, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productAll(matrix) {
      let v = 1;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v *= matrix.get(i, j);
        }
      }
      return v;
    }
    function varianceByRow(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let i = 0; i < rows; i++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean[i];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / cols) / cols);
        }
      }
      return variance;
    }
    function varianceByColumn(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let j = 0; j < cols; j++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let i = 0; i < rows; i++) {
          x = matrix.get(i, j) - mean[j];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / rows) / rows);
        }
      }
      return variance;
    }
    function varianceAll(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const size = rows * cols;
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean;
          sum1 += x;
          sum2 += x * x;
        }
      }
      if (unbiased) {
        return (sum2 - sum1 * sum1 / size) / (size - 1);
      } else {
        return (sum2 - sum1 * sum1 / size) / size;
      }
    }
    function centerByRow(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[i]);
        }
      }
    }
    function centerByColumn(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[j]);
        }
      }
    }
    function centerAll(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean);
        }
      }
    }
    function getScaleByRow(matrix) {
      const scale = [];
      for (let i = 0; i < matrix.rows; i++) {
        let sum = 0;
        for (let j = 0; j < matrix.columns; j++) {
          sum += matrix.get(i, j) ** 2 / (matrix.columns - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByRow(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[i]);
        }
      }
    }
    function getScaleByColumn(matrix) {
      const scale = [];
      for (let j = 0; j < matrix.columns; j++) {
        let sum = 0;
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / (matrix.rows - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByColumn(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[j]);
        }
      }
    }
    function getScaleAll(matrix) {
      const divider = matrix.size - 1;
      let sum = 0;
      for (let j = 0; j < matrix.columns; j++) {
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / divider;
        }
      }
      return Math.sqrt(sum);
    }
    function scaleAll(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale);
        }
      }
    }
    class AbstractMatrix {
      static from1DArray(newRows, newColumns, newData) {
        let length = newRows * newColumns;
        if (length !== newData.length) {
          throw new RangeError('data length does not match given dimensions');
        }
        let newMatrix = new Matrix$1(newRows, newColumns);
        for (let row = 0; row < newRows; row++) {
          for (let column = 0; column < newColumns; column++) {
            newMatrix.set(row, column, newData[row * newColumns + column]);
          }
        }
        return newMatrix;
      }
      static rowVector(newData) {
        let vector = new Matrix$1(1, newData.length);
        for (let i = 0; i < newData.length; i++) {
          vector.set(0, i, newData[i]);
        }
        return vector;
      }
      static columnVector(newData) {
        let vector = new Matrix$1(newData.length, 1);
        for (let i = 0; i < newData.length; i++) {
          vector.set(i, 0, newData[i]);
        }
        return vector;
      }
      static zeros(rows, columns) {
        return new Matrix$1(rows, columns);
      }
      static ones(rows, columns) {
        return new Matrix$1(rows, columns).fill(1);
      }
      static rand(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          random = Math.random
        } = options;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.set(i, j, random());
          }
        }
        return matrix;
      }
      static randInt(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1000,
          random = Math.random
        } = options;
        if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
        if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let interval = max - min;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            let value = min + Math.round(random() * interval);
            matrix.set(i, j, value);
          }
        }
        return matrix;
      }
      static eye(rows, columns, value) {
        if (columns === undefined) columns = rows;
        if (value === undefined) value = 1;
        let min = Math.min(rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, value);
        }
        return matrix;
      }
      static diag(data, rows, columns) {
        let l = data.length;
        if (rows === undefined) rows = l;
        if (columns === undefined) columns = rows;
        let min = Math.min(l, rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, data[i]);
        }
        return matrix;
      }
      static min(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static max(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new this(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static checkMatrix(value) {
        return AbstractMatrix.isMatrix(value) ? value : new Matrix$1(value);
      }
      static isMatrix(value) {
        return value != null && value.klass === 'Matrix';
      }
      get size() {
        return this.rows * this.columns;
      }
      apply(callback) {
        if (typeof callback !== 'function') {
          throw new TypeError('callback must be a function');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            callback.call(this, i, j);
          }
        }
        return this;
      }
      to1DArray() {
        let array = [];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            array.push(this.get(i, j));
          }
        }
        return array;
      }
      to2DArray() {
        let copy = [];
        for (let i = 0; i < this.rows; i++) {
          copy.push([]);
          for (let j = 0; j < this.columns; j++) {
            copy[i].push(this.get(i, j));
          }
        }
        return copy;
      }
      toJSON() {
        return this.to2DArray();
      }
      isRowVector() {
        return this.rows === 1;
      }
      isColumnVector() {
        return this.columns === 1;
      }
      isVector() {
        return this.rows === 1 || this.columns === 1;
      }
      isSquare() {
        return this.rows === this.columns;
      }
      isEmpty() {
        return this.rows === 0 || this.columns === 0;
      }
      isSymmetric() {
        if (this.isSquare()) {
          for (let i = 0; i < this.rows; i++) {
            for (let j = 0; j <= i; j++) {
              if (this.get(i, j) !== this.get(j, i)) {
                return false;
              }
            }
          }
          return true;
        }
        return false;
      }
      isDistance() {
        if (!this.isSymmetric()) return false;
        for (let i = 0; i < this.rows; i++) {
          if (this.get(i, i) !== 0) return false;
        }
        return true;
      }
      isEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isEchelonForm = true;
        let checked = false;
        while (i < this.rows && isEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isEchelonForm = false;
              checked = true;
            }
          }
          i++;
        }
        return isEchelonForm;
      }
      isReducedEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isReducedEchelonForm = true;
        let checked = false;
        while (i < this.rows && isReducedEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isReducedEchelonForm = false;
              checked = true;
            }
          }
          for (let k = j + 1; k < this.rows; k++) {
            if (this.get(i, k) !== 0) {
              isReducedEchelonForm = false;
            }
          }
          i++;
        }
        return isReducedEchelonForm;
      }
      echelonForm() {
        let result = this.clone();
        let h = 0;
        let k = 0;
        while (h < result.rows && k < result.columns) {
          let iMax = h;
          for (let i = h; i < result.rows; i++) {
            if (result.get(i, k) > result.get(iMax, k)) {
              iMax = i;
            }
          }
          if (result.get(iMax, k) === 0) {
            k++;
          } else {
            result.swapRows(h, iMax);
            let tmp = result.get(h, k);
            for (let j = k; j < result.columns; j++) {
              result.set(h, j, result.get(h, j) / tmp);
            }
            for (let i = h + 1; i < result.rows; i++) {
              let factor = result.get(i, k) / result.get(h, k);
              result.set(i, k, 0);
              for (let j = k + 1; j < result.columns; j++) {
                result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
              }
            }
            h++;
            k++;
          }
        }
        return result;
      }
      reducedEchelonForm() {
        let result = this.echelonForm();
        let m = result.columns;
        let n = result.rows;
        let h = n - 1;
        while (h >= 0) {
          if (result.maxRow(h) === 0) {
            h--;
          } else {
            let p = 0;
            let pivot = false;
            while (p < n && pivot === false) {
              if (result.get(h, p) === 1) {
                pivot = true;
              } else {
                p++;
              }
            }
            for (let i = 0; i < h; i++) {
              let factor = result.get(i, p);
              for (let j = p; j < m; j++) {
                let tmp = result.get(i, j) - factor * result.get(h, j);
                result.set(i, j, tmp);
              }
            }
            h--;
          }
        }
        return result;
      }
      set() {
        throw new Error('set method is unimplemented');
      }
      get() {
        throw new Error('get method is unimplemented');
      }
      repeat(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          rows = 1,
          columns = 1
        } = options;
        if (!Number.isInteger(rows) || rows <= 0) {
          throw new TypeError('rows must be a positive integer');
        }
        if (!Number.isInteger(columns) || columns <= 0) {
          throw new TypeError('columns must be a positive integer');
        }
        let matrix = new Matrix$1(this.rows * rows, this.columns * columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.setSubMatrix(this, this.rows * i, this.columns * j);
          }
        }
        return matrix;
      }
      fill(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, value);
          }
        }
        return this;
      }
      neg() {
        return this.mulS(-1);
      }
      getRow(index) {
        checkRowIndex(this, index);
        let row = [];
        for (let i = 0; i < this.columns; i++) {
          row.push(this.get(index, i));
        }
        return row;
      }
      getRowVector(index) {
        return Matrix$1.rowVector(this.getRow(index));
      }
      setRow(index, array) {
        checkRowIndex(this, index);
        array = checkRowVector(this, array);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, array[i]);
        }
        return this;
      }
      swapRows(row1, row2) {
        checkRowIndex(this, row1);
        checkRowIndex(this, row2);
        for (let i = 0; i < this.columns; i++) {
          let temp = this.get(row1, i);
          this.set(row1, i, this.get(row2, i));
          this.set(row2, i, temp);
        }
        return this;
      }
      getColumn(index) {
        checkColumnIndex(this, index);
        let column = [];
        for (let i = 0; i < this.rows; i++) {
          column.push(this.get(i, index));
        }
        return column;
      }
      getColumnVector(index) {
        return Matrix$1.columnVector(this.getColumn(index));
      }
      setColumn(index, array) {
        checkColumnIndex(this, index);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, array[i]);
        }
        return this;
      }
      swapColumns(column1, column2) {
        checkColumnIndex(this, column1);
        checkColumnIndex(this, column2);
        for (let i = 0; i < this.rows; i++) {
          let temp = this.get(i, column1);
          this.set(i, column1, this.get(i, column2));
          this.set(i, column2, temp);
        }
        return this;
      }
      addRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[j]);
          }
        }
        return this;
      }
      subRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[j]);
          }
        }
        return this;
      }
      mulRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[j]);
          }
        }
        return this;
      }
      divRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[j]);
          }
        }
        return this;
      }
      addColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[i]);
          }
        }
        return this;
      }
      subColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[i]);
          }
        }
        return this;
      }
      mulColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[i]);
          }
        }
        return this;
      }
      divColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[i]);
          }
        }
        return this;
      }
      mulRow(index, value) {
        checkRowIndex(this, index);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, this.get(index, i) * value);
        }
        return this;
      }
      mulColumn(index, value) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, this.get(i, index) * value);
        }
        return this;
      }
      max(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[row]) {
                    max[row] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case 'column':
            {
              const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[column]) {
                    max[column] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case undefined:
            {
              let max = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max) {
                    max = this.get(row, column);
                  }
                }
              }
              return max;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      maxIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) > v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      min(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[row]) {
                    min[row] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case 'column':
            {
              const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[column]) {
                    min[column] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case undefined:
            {
              let min = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min) {
                    min = this.get(row, column);
                  }
                }
              }
              return min;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      minIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) < v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      maxRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      maxRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      minRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      minRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      maxColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      maxColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      minColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      minColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      diag() {
        let min = Math.min(this.rows, this.columns);
        let diag = [];
        for (let i = 0; i < min; i++) {
          diag.push(this.get(i, i));
        }
        return diag;
      }
      norm(type = 'frobenius') {
        switch (type) {
          case 'max':
            return this.max();
          case 'frobenius':
            return Math.sqrt(this.dot(this));
          default:
            throw new RangeError(`unknown norm type: ${type}`);
        }
      }
      cumulativeSum() {
        let sum = 0;
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            sum += this.get(i, j);
            this.set(i, j, sum);
          }
        }
        return this;
      }
      dot(vector2) {
        if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
        let vector1 = this.to1DArray();
        if (vector1.length !== vector2.length) {
          throw new RangeError('vectors do not have the same size');
        }
        let dot = 0;
        for (let i = 0; i < vector1.length; i++) {
          dot += vector1[i] * vector2[i];
        }
        return dot;
      }
      mmul(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.columns;
        let result = new Matrix$1(m, p);
        let Bcolj = new Float64Array(n);
        for (let j = 0; j < p; j++) {
          for (let k = 0; k < n; k++) {
            Bcolj[k] = other.get(k, j);
          }
          for (let i = 0; i < m; i++) {
            let s = 0;
            for (let k = 0; k < n; k++) {
              s += this.get(i, k) * Bcolj[k];
            }
            result.set(i, j, s);
          }
        }
        return result;
      }
      mpow(scalar) {
        if (!this.isSquare()) {
          throw new RangeError('Matrix must be square');
        }
        if (!Number.isInteger(scalar) || scalar < 0) {
          throw new RangeError('Exponent must be a non-negative integer');
        }
        // Russian Peasant exponentiation, i.e. exponentiation by squaring
        let result = Matrix$1.eye(this.rows);
        let bb = this;
        // Note: Don't bit shift. In JS, that would truncate at 32 bits
        for (let e = scalar; e >= 1; e /= 2) {
          if ((e & 1) !== 0) {
            result = result.mmul(bb);
          }
          bb = bb.mmul(bb);
        }
        return result;
      }
      strassen2x2(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(2, 2);
        const a11 = this.get(0, 0);
        const b11 = other.get(0, 0);
        const a12 = this.get(0, 1);
        const b12 = other.get(0, 1);
        const a21 = this.get(1, 0);
        const b21 = other.get(1, 0);
        const a22 = this.get(1, 1);
        const b22 = other.get(1, 1);

        // Compute intermediate values.
        const m1 = (a11 + a22) * (b11 + b22);
        const m2 = (a21 + a22) * b11;
        const m3 = a11 * (b12 - b22);
        const m4 = a22 * (b21 - b11);
        const m5 = (a11 + a12) * b22;
        const m6 = (a21 - a11) * (b11 + b12);
        const m7 = (a12 - a22) * (b21 + b22);

        // Combine intermediate values into the output.
        const c00 = m1 + m4 - m5 + m7;
        const c01 = m3 + m5;
        const c10 = m2 + m4;
        const c11 = m1 - m2 + m3 + m6;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        return result;
      }
      strassen3x3(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(3, 3);
        const a00 = this.get(0, 0);
        const a01 = this.get(0, 1);
        const a02 = this.get(0, 2);
        const a10 = this.get(1, 0);
        const a11 = this.get(1, 1);
        const a12 = this.get(1, 2);
        const a20 = this.get(2, 0);
        const a21 = this.get(2, 1);
        const a22 = this.get(2, 2);
        const b00 = other.get(0, 0);
        const b01 = other.get(0, 1);
        const b02 = other.get(0, 2);
        const b10 = other.get(1, 0);
        const b11 = other.get(1, 1);
        const b12 = other.get(1, 2);
        const b20 = other.get(2, 0);
        const b21 = other.get(2, 1);
        const b22 = other.get(2, 2);
        const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
        const m2 = (a00 - a10) * (-b01 + b11);
        const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
        const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
        const m5 = (a10 + a11) * (-b00 + b01);
        const m6 = a00 * b00;
        const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
        const m8 = (-a00 + a20) * (b02 - b12);
        const m9 = (a20 + a21) * (-b00 + b02);
        const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
        const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
        const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
        const m13 = (a02 - a22) * (b11 - b21);
        const m14 = a02 * b20;
        const m15 = (a21 + a22) * (-b20 + b21);
        const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
        const m17 = (a02 - a12) * (b12 - b22);
        const m18 = (a11 + a12) * (-b20 + b22);
        const m19 = a01 * b10;
        const m20 = a12 * b21;
        const m21 = a10 * b02;
        const m22 = a20 * b01;
        const m23 = a22 * b22;
        const c00 = m6 + m14 + m19;
        const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
        const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
        const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
        const c11 = m2 + m4 + m5 + m6 + m20;
        const c12 = m14 + m16 + m17 + m18 + m21;
        const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
        const c21 = m12 + m13 + m14 + m15 + m22;
        const c22 = m6 + m7 + m8 + m9 + m23;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(0, 2, c02);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        result.set(1, 2, c12);
        result.set(2, 0, c20);
        result.set(2, 1, c21);
        result.set(2, 2, c22);
        return result;
      }
      mmulStrassen(y) {
        y = Matrix$1.checkMatrix(y);
        let x = this.clone();
        let r1 = x.rows;
        let c1 = x.columns;
        let r2 = y.rows;
        let c2 = y.columns;
        if (c1 !== r2) {
          // eslint-disable-next-line no-console
          console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
        }

        // Put a matrix into the top left of a matrix of zeros.
        // `rows` and `cols` are the dimensions of the output matrix.
        function embed(mat, rows, cols) {
          let r = mat.rows;
          let c = mat.columns;
          if (r === rows && c === cols) {
            return mat;
          } else {
            let resultat = AbstractMatrix.zeros(rows, cols);
            resultat = resultat.setSubMatrix(mat, 0, 0);
            return resultat;
          }
        }

        // Make sure both matrices are the same size.
        // This is exclusively for simplicity:
        // this algorithm can be implemented with matrices of different sizes.

        let r = Math.max(r1, r2);
        let c = Math.max(c1, c2);
        x = embed(x, r, c);
        y = embed(y, r, c);

        // Our recursive multiplication function.
        function blockMult(a, b, rows, cols) {
          // For small matrices, resort to naive multiplication.
          if (rows <= 512 || cols <= 512) {
            return a.mmul(b); // a is equivalent to this
          }

          // Apply dynamic padding.
          if (rows % 2 === 1 && cols % 2 === 1) {
            a = embed(a, rows + 1, cols + 1);
            b = embed(b, rows + 1, cols + 1);
          } else if (rows % 2 === 1) {
            a = embed(a, rows + 1, cols);
            b = embed(b, rows + 1, cols);
          } else if (cols % 2 === 1) {
            a = embed(a, rows, cols + 1);
            b = embed(b, rows, cols + 1);
          }
          let halfRows = parseInt(a.rows / 2, 10);
          let halfCols = parseInt(a.columns / 2, 10);
          // Subdivide input matrices.
          let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
          let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
          let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
          let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
          let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
          let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

          // Compute intermediate values.
          let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
          let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
          let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
          let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
          let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
          let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
          let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

          // Combine intermediate values into the output.
          let c11 = AbstractMatrix.add(m1, m4);
          c11.sub(m5);
          c11.add(m7);
          let c12 = AbstractMatrix.add(m3, m5);
          let c21 = AbstractMatrix.add(m2, m4);
          let c22 = AbstractMatrix.sub(m1, m2);
          c22.add(m3);
          c22.add(m6);

          // Crop output to the desired size (undo dynamic padding).
          let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
          result = result.setSubMatrix(c11, 0, 0);
          result = result.setSubMatrix(c12, c11.rows, 0);
          result = result.setSubMatrix(c21, 0, c11.columns);
          result = result.setSubMatrix(c22, c11.rows, c11.columns);
          return result.subMatrix(0, rows - 1, 0, cols - 1);
        }
        return blockMult(x, y, r, c);
      }
      scaleRows(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.rows; i++) {
          const row = this.getRow(i);
          if (row.length > 0) {
            rescale(row, {
              min,
              max,
              output: row
            });
          }
          newMatrix.setRow(i, row);
        }
        return newMatrix;
      }
      scaleColumns(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.columns; i++) {
          const column = this.getColumn(i);
          if (column.length) {
            rescale(column, {
              min,
              max,
              output: column
            });
          }
          newMatrix.setColumn(i, column);
        }
        return newMatrix;
      }
      flipRows() {
        const middle = Math.ceil(this.columns / 2);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < middle; j++) {
            let first = this.get(i, j);
            let last = this.get(i, this.columns - 1 - j);
            this.set(i, j, last);
            this.set(i, this.columns - 1 - j, first);
          }
        }
        return this;
      }
      flipColumns() {
        const middle = Math.ceil(this.rows / 2);
        for (let j = 0; j < this.columns; j++) {
          for (let i = 0; i < middle; i++) {
            let first = this.get(i, j);
            let last = this.get(this.rows - 1 - i, j);
            this.set(i, j, last);
            this.set(this.rows - 1 - i, j, first);
          }
        }
        return this;
      }
      kroneckerProduct(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.rows;
        let q = other.columns;
        let result = new Matrix$1(m * p, n * q);
        for (let i = 0; i < m; i++) {
          for (let j = 0; j < n; j++) {
            for (let k = 0; k < p; k++) {
              for (let l = 0; l < q; l++) {
                result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
              }
            }
          }
        }
        return result;
      }
      kroneckerSum(other) {
        other = Matrix$1.checkMatrix(other);
        if (!this.isSquare() || !other.isSquare()) {
          throw new Error('Kronecker Sum needs two Square Matrices');
        }
        let m = this.rows;
        let n = other.rows;
        let AxI = this.kroneckerProduct(Matrix$1.eye(n, n));
        let IxB = Matrix$1.eye(m, m).kroneckerProduct(other);
        return AxI.add(IxB);
      }
      transpose() {
        let result = new Matrix$1(this.columns, this.rows);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result.set(j, i, this.get(i, j));
          }
        }
        return result;
      }
      sortRows(compareFunction = compareNumbers) {
        for (let i = 0; i < this.rows; i++) {
          this.setRow(i, this.getRow(i).sort(compareFunction));
        }
        return this;
      }
      sortColumns(compareFunction = compareNumbers) {
        for (let i = 0; i < this.columns; i++) {
          this.setColumn(i, this.getColumn(i).sort(compareFunction));
        }
        return this;
      }
      subMatrix(startRow, endRow, startColumn, endColumn) {
        checkRange(this, startRow, endRow, startColumn, endColumn);
        let newMatrix = new Matrix$1(endRow - startRow + 1, endColumn - startColumn + 1);
        for (let i = startRow; i <= endRow; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
          }
        }
        return newMatrix;
      }
      subMatrixRow(indices, startColumn, endColumn) {
        if (startColumn === undefined) startColumn = 0;
        if (endColumn === undefined) endColumn = this.columns - 1;
        if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(indices.length, endColumn - startColumn + 1);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            if (indices[i] < 0 || indices[i] >= this.rows) {
              throw new RangeError(`Row index out of range: ${indices[i]}`);
            }
            newMatrix.set(i, j - startColumn, this.get(indices[i], j));
          }
        }
        return newMatrix;
      }
      subMatrixColumn(indices, startRow, endRow) {
        if (startRow === undefined) startRow = 0;
        if (endRow === undefined) endRow = this.rows - 1;
        if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(endRow - startRow + 1, indices.length);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startRow; j <= endRow; j++) {
            if (indices[i] < 0 || indices[i] >= this.columns) {
              throw new RangeError(`Column index out of range: ${indices[i]}`);
            }
            newMatrix.set(j - startRow, i, this.get(j, indices[i]));
          }
        }
        return newMatrix;
      }
      setSubMatrix(matrix, startRow, startColumn) {
        matrix = Matrix$1.checkMatrix(matrix);
        if (matrix.isEmpty()) {
          return this;
        }
        let endRow = startRow + matrix.rows - 1;
        let endColumn = startColumn + matrix.columns - 1;
        checkRange(this, startRow, endRow, startColumn, endColumn);
        for (let i = 0; i < matrix.rows; i++) {
          for (let j = 0; j < matrix.columns; j++) {
            this.set(startRow + i, startColumn + j, matrix.get(i, j));
          }
        }
        return this;
      }
      selection(rowIndices, columnIndices) {
        checkRowIndices(this, rowIndices);
        checkColumnIndices(this, columnIndices);
        let newMatrix = new Matrix$1(rowIndices.length, columnIndices.length);
        for (let i = 0; i < rowIndices.length; i++) {
          let rowIndex = rowIndices[i];
          for (let j = 0; j < columnIndices.length; j++) {
            let columnIndex = columnIndices[j];
            newMatrix.set(i, j, this.get(rowIndex, columnIndex));
          }
        }
        return newMatrix;
      }
      trace() {
        let min = Math.min(this.rows, this.columns);
        let trace = 0;
        for (let i = 0; i < min; i++) {
          trace += this.get(i, i);
        }
        return trace;
      }
      clone() {
        return this.constructor.copy(this, new Matrix$1(this.rows, this.columns));
      }

      /**
       * @template {AbstractMatrix} M
       * @param {AbstractMatrix} from
       * @param {M} to
       * @return {M}
       */
      static copy(from, to) {
        for (const [row, column, value] of from.entries()) {
          to.set(row, column, value);
        }
        return to;
      }
      sum(by) {
        switch (by) {
          case 'row':
            return sumByRow(this);
          case 'column':
            return sumByColumn(this);
          case undefined:
            return sumAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      product(by) {
        switch (by) {
          case 'row':
            return productByRow(this);
          case 'column':
            return productByColumn(this);
          case undefined:
            return productAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      mean(by) {
        const sum = this.sum(by);
        switch (by) {
          case 'row':
            {
              for (let i = 0; i < this.rows; i++) {
                sum[i] /= this.columns;
              }
              return sum;
            }
          case 'column':
            {
              for (let i = 0; i < this.columns; i++) {
                sum[i] /= this.rows;
              }
              return sum;
            }
          case undefined:
            return sum / this.size;
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      variance(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          unbiased = true,
          mean = this.mean(by)
        } = options;
        if (typeof unbiased !== 'boolean') {
          throw new TypeError('unbiased must be a boolean');
        }
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByRow(this, unbiased, mean);
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByColumn(this, unbiased, mean);
            }
          case undefined:
            {
              if (typeof mean !== 'number') {
                throw new TypeError('mean must be a number');
              }
              return varianceAll(this, unbiased, mean);
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      standardDeviation(by, options) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        const variance = this.variance(by, options);
        if (by === undefined) {
          return Math.sqrt(variance);
        } else {
          for (let i = 0; i < variance.length; i++) {
            variance[i] = Math.sqrt(variance[i]);
          }
          return variance;
        }
      }
      center(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          center = this.mean(by)
        } = options;
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByRow(this, center);
              return this;
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByColumn(this, center);
              return this;
            }
          case undefined:
            {
              if (typeof center !== 'number') {
                throw new TypeError('center must be a number');
              }
              centerAll(this, center);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      scale(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        let scale = options.scale;
        switch (by) {
          case 'row':
            {
              if (scale === undefined) {
                scale = getScaleByRow(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByRow(this, scale);
              return this;
            }
          case 'column':
            {
              if (scale === undefined) {
                scale = getScaleByColumn(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByColumn(this, scale);
              return this;
            }
          case undefined:
            {
              if (scale === undefined) {
                scale = getScaleAll(this);
              } else if (typeof scale !== 'number') {
                throw new TypeError('scale must be a number');
              }
              scaleAll(this, scale);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      toString(options) {
        return inspectMatrixWithOptions(this, options);
      }
      [Symbol.iterator]() {
        return this.entries();
      }

      /**
       * iterator from left to right, from top to bottom
       * yield [row, column, value]
       * @returns {Generator<[number, number, number], void, void>}
       */
      *entries() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield [row, col, this.get(row, col)];
          }
        }
      }

      /**
       * iterator from left to right, from top to bottom
       * yield value
       * @returns {Generator<number, void, void>}
       */
      *values() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield this.get(row, col);
          }
        }
      }
    }
    AbstractMatrix.prototype.klass = 'Matrix';
    if (typeof Symbol !== 'undefined') {
      AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
    }
    function compareNumbers(a, b) {
      return a - b;
    }
    function isArrayOfNumbers(array) {
      return array.every(element => {
        return typeof element === 'number';
      });
    }

    // Synonyms
    AbstractMatrix.random = AbstractMatrix.rand;
    AbstractMatrix.randomInt = AbstractMatrix.randInt;
    AbstractMatrix.diagonal = AbstractMatrix.diag;
    AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
    AbstractMatrix.identity = AbstractMatrix.eye;
    AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
    AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
    let Matrix$1 = class Matrix extends AbstractMatrix {
      /**
       * @type {Float64Array[]}
       */
      data;

      /**
       * Init an empty matrix
       * @param {number} nRows
       * @param {number} nColumns
       */
      #initData(nRows, nColumns) {
        this.data = [];
        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
        this.rows = nRows;
        this.columns = nColumns;
      }
      constructor(nRows, nColumns) {
        super();
        if (Matrix.isMatrix(nRows)) {
          this.#initData(nRows.rows, nRows.columns);
          Matrix.copy(nRows, this);
        } else if (Number.isInteger(nRows) && nRows >= 0) {
          this.#initData(nRows, nColumns);
        } else if (isAnyArray.isAnyArray(nRows)) {
          // Copy the values from the 2D array
          const arrayData = nRows;
          nRows = arrayData.length;
          nColumns = nRows ? arrayData[0].length : 0;
          if (typeof nColumns !== 'number') {
            throw new TypeError('Data must be a 2D array with at least one element');
          }
          this.data = [];
          for (let i = 0; i < nRows; i++) {
            if (arrayData[i].length !== nColumns) {
              throw new RangeError('Inconsistent array dimensions');
            }
            if (!isArrayOfNumbers(arrayData[i])) {
              throw new TypeError('Input data contains non-numeric values');
            }
            this.data.push(Float64Array.from(arrayData[i]));
          }
          this.rows = nRows;
          this.columns = nColumns;
        } else {
          throw new TypeError('First argument must be a positive number or an array');
        }
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
      removeRow(index) {
        checkRowIndex(this, index);
        this.data.splice(index, 1);
        this.rows -= 1;
        return this;
      }
      addRow(index, array) {
        if (array === undefined) {
          array = index;
          index = this.rows;
        }
        checkRowIndex(this, index, true);
        array = Float64Array.from(checkRowVector(this, array));
        this.data.splice(index, 0, array);
        this.rows += 1;
        return this;
      }
      removeColumn(index) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns - 1);
          for (let j = 0; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          for (let j = index + 1; j < this.columns; j++) {
            newRow[j - 1] = this.data[i][j];
          }
          this.data[i] = newRow;
        }
        this.columns -= 1;
        return this;
      }
      addColumn(index, array) {
        if (typeof array === 'undefined') {
          array = index;
          index = this.columns;
        }
        checkColumnIndex(this, index, true);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns + 1);
          let j = 0;
          for (; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          newRow[j++] = array[i];
          for (; j < this.columns + 1; j++) {
            newRow[j] = this.data[i][j - 1];
          }
          this.data[i] = newRow;
        }
        this.columns += 1;
        return this;
      }
    };
    installMathOperations(AbstractMatrix, Matrix$1);

    /**
     * @typedef {0 | 1 | number | boolean} Mask
     */

    class SymmetricMatrix extends AbstractMatrix {
      /** @type {Matrix} */
      #matrix;
      get size() {
        return this.#matrix.size;
      }
      get rows() {
        return this.#matrix.rows;
      }
      get columns() {
        return this.#matrix.columns;
      }
      get diagonalSize() {
        return this.rows;
      }

      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isSymmetricMatrix(value) {
        return Matrix$1.isMatrix(value) && value.klassType === 'SymmetricMatrix';
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static zeros(diagonalSize) {
        return new this(diagonalSize);
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static ones(diagonalSize) {
        return new this(diagonalSize).fill(1);
      }

      /**
       * @param {number | AbstractMatrix | ArrayLike<ArrayLike<number>>} diagonalSize
       * @return {this}
       */
      constructor(diagonalSize) {
        super();
        if (Matrix$1.isMatrix(diagonalSize)) {
          if (!diagonalSize.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
          this.#matrix = Matrix$1.copy(diagonalSize, new Matrix$1(diagonalSize.rows, diagonalSize.rows));
        } else if (Number.isInteger(diagonalSize) && diagonalSize >= 0) {
          this.#matrix = new Matrix$1(diagonalSize, diagonalSize);
        } else {
          this.#matrix = new Matrix$1(diagonalSize);
          if (!this.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
        }
      }
      clone() {
        const matrix = new SymmetricMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          matrix.set(row, col, value);
        }
        return matrix;
      }
      toMatrix() {
        return new Matrix$1(this);
      }
      get(rowIndex, columnIndex) {
        return this.#matrix.get(rowIndex, columnIndex);
      }
      set(rowIndex, columnIndex, value) {
        // symmetric set
        this.#matrix.set(rowIndex, columnIndex, value);
        this.#matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      removeCross(index) {
        // symmetric remove side
        this.#matrix.removeRow(index);
        this.#matrix.removeColumn(index);
        return this;
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }
        const row = array.slice();
        row.splice(index, 1);
        this.#matrix.addRow(index, row);
        this.#matrix.addColumn(index, array);
        return this;
      }

      /**
       * @param {Mask[]} mask
       */
      applyMask(mask) {
        if (mask.length !== this.diagonalSize) {
          throw new RangeError('Mask size do not match with matrix size');
        }

        // prepare sides to remove from matrix from mask
        /** @type {number[]} */
        const sidesToRemove = [];
        for (const [index, passthroughs] of mask.entries()) {
          if (passthroughs) continue;
          sidesToRemove.push(index);
        }
        // to remove from highest to lowest for no mutation shifting
        sidesToRemove.reverse();

        // remove sides
        for (const sideIndex of sidesToRemove) {
          this.removeCross(sideIndex);
        }
        return this;
      }

      /**
       * Compact format upper-right corner of matrix
       * iterate from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 1 2 3 4
       * B 2 5 6 7
       * C 3 6 8 9
       * D 4 7 9 10
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]`
       *
       * length is S(i=0, n=sideSize) => 10 for a 4 sideSized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;

        /** @type {number[]} */
        const compact = new Array(diagonalSize * (diagonalSize + 1) / 2);
        for (let col = 0, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       * @return {SymmetricMatrix}
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        // compactSize = (sideSize * (sideSize + 1)) / 2
        // https://mathsolver.microsoft.com/fr/solve-problem/y%20%3D%20%20x%20%60cdot%20%20%20%60frac%7B%20%20%60left(%20x%2B1%20%20%60right)%20%20%20%20%7D%7B%202%20%20%7D
        // sideSize = (Sqrt(8 × compactSize + 1) - 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) - 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a Symmetric Matrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new SymmetricMatrix(diagonalSize);
        for (let col = 0, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row;
        }
        return matrix;
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield [row, column, value]
       *
       * @returns {Generator<[number, number, number], void, void>}
       */
      *upperRightEntries() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield [row, col, value];

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield value
       *
       * @returns {Generator<[number, number, number], void, void>}
       */
      *upperRightValues() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield value;

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }
    }
    SymmetricMatrix.prototype.klassType = 'SymmetricMatrix';
    class DistanceMatrix extends SymmetricMatrix {
      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isDistanceMatrix(value) {
        return SymmetricMatrix.isSymmetricMatrix(value) && value.klassSubType === 'DistanceMatrix';
      }
      constructor(sideSize) {
        super(sideSize);
        if (!this.isDistance()) {
          throw new TypeError('Provided arguments do no produce a distance matrix');
        }
      }
      set(rowIndex, columnIndex, value) {
        // distance matrix diagonal is 0
        if (rowIndex === columnIndex) value = 0;
        return super.set(rowIndex, columnIndex, value);
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }

        // ensure distance
        array = array.slice();
        array[index] = 0;
        return super.addCross(index, array);
      }
      toSymmetricMatrix() {
        return new SymmetricMatrix(this);
      }
      clone() {
        const matrix = new DistanceMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          if (row === col) continue;
          matrix.set(row, col, value);
        }
        return matrix;
      }

      /**
       * Compact format upper-right corner of matrix
       * no diagonal (only zeros)
       * iterable from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 0 1 2 3
       * B 1 0 4 5
       * C 2 4 0 6
       * D 3 5 6 0
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6]`
       *
       * length is S(i=0, n=sideSize-1) => 6 for a 4 side sized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;
        const compactLength = (diagonalSize - 1) * diagonalSize / 2;

        /** @type {number[]} */
        const compact = new Array(compactLength);
        for (let col = 1, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        if (compactSize === 0) {
          return new this(0);
        }

        // compactSize in Natural integer range ]0;∞]
        // compactSize = (sideSize * (sideSize - 1)) / 2
        // sideSize = (Sqrt(8 × compactSize + 1) + 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) + 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a DistanceMatrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new this(diagonalSize);
        for (let col = 1, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return matrix;
      }
    }
    DistanceMatrix.prototype.klassSubType = 'DistanceMatrix';
    class BaseView extends AbstractMatrix {
      constructor(matrix, rows, columns) {
        super();
        this.matrix = matrix;
        this.rows = rows;
        this.columns = columns;
      }
    }
    class MatrixColumnView extends BaseView {
      constructor(matrix, column) {
        checkColumnIndex(matrix, column);
        super(matrix, matrix.rows, 1);
        this.column = column;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.column, value);
        return this;
      }
      get(rowIndex) {
        return this.matrix.get(rowIndex, this.column);
      }
    }
    class MatrixColumnSelectionView extends BaseView {
      constructor(matrix, columnIndices) {
        checkColumnIndices(matrix, columnIndices);
        super(matrix, matrix.rows, columnIndices.length);
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
      }
    }
    class MatrixFlipColumnView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
      }
    }
    class MatrixFlipRowView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
      }
    }
    class MatrixRowView extends BaseView {
      constructor(matrix, row) {
        checkRowIndex(matrix, row);
        super(matrix, 1, matrix.columns);
        this.row = row;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.row, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.row, columnIndex);
      }
    }
    class MatrixRowSelectionView extends BaseView {
      constructor(matrix, rowIndices) {
        checkRowIndices(matrix, rowIndices);
        super(matrix, rowIndices.length, matrix.columns);
        this.rowIndices = rowIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
      }
    }
    class MatrixSelectionView extends BaseView {
      constructor(matrix, rowIndices, columnIndices) {
        checkRowIndices(matrix, rowIndices);
        checkColumnIndices(matrix, columnIndices);
        super(matrix, rowIndices.length, columnIndices.length);
        this.rowIndices = rowIndices;
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
      }
    }
    class MatrixSubView extends BaseView {
      constructor(matrix, startRow, endRow, startColumn, endColumn) {
        checkRange(matrix, startRow, endRow, startColumn, endColumn);
        super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
        this.startRow = startRow;
        this.startColumn = startColumn;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
      }
    }
    class MatrixTransposeView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.columns, matrix.rows);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(columnIndex, rowIndex);
      }
    }
    class WrapperMatrix1D extends AbstractMatrix {
      constructor(data, options = {}) {
        const {
          rows = 1
        } = options;
        if (data.length % rows !== 0) {
          throw new Error('the data length is not divisible by the number of rows');
        }
        super();
        this.rows = rows;
        this.columns = data.length / rows;
        this.data = data;
      }
      set(rowIndex, columnIndex, value) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        this.data[index] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        return this.data[index];
      }
      _calculateIndex(row, column) {
        return row * this.columns + column;
      }
    }
    class WrapperMatrix2D extends AbstractMatrix {
      constructor(data) {
        super();
        this.data = data;
        this.rows = data.length;
        this.columns = data[0].length;
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
    }
    function wrap(array, options) {
      if (isAnyArray.isAnyArray(array)) {
        if (array[0] && isAnyArray.isAnyArray(array[0])) {
          return new WrapperMatrix2D(array);
        } else {
          return new WrapperMatrix1D(array, options);
        }
      } else {
        throw new Error('the argument is not an array');
      }
    }
    class LuDecomposition {
      constructor(matrix) {
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        let lu = matrix.clone();
        let rows = lu.rows;
        let columns = lu.columns;
        let pivotVector = new Float64Array(rows);
        let pivotSign = 1;
        let i, j, k, p, s, t, v;
        let LUcolj, kmax;
        for (i = 0; i < rows; i++) {
          pivotVector[i] = i;
        }
        LUcolj = new Float64Array(rows);
        for (j = 0; j < columns; j++) {
          for (i = 0; i < rows; i++) {
            LUcolj[i] = lu.get(i, j);
          }
          for (i = 0; i < rows; i++) {
            kmax = Math.min(i, j);
            s = 0;
            for (k = 0; k < kmax; k++) {
              s += lu.get(i, k) * LUcolj[k];
            }
            LUcolj[i] -= s;
            lu.set(i, j, LUcolj[i]);
          }
          p = j;
          for (i = j + 1; i < rows; i++) {
            if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
              p = i;
            }
          }
          if (p !== j) {
            for (k = 0; k < columns; k++) {
              t = lu.get(p, k);
              lu.set(p, k, lu.get(j, k));
              lu.set(j, k, t);
            }
            v = pivotVector[p];
            pivotVector[p] = pivotVector[j];
            pivotVector[j] = v;
            pivotSign = -pivotSign;
          }
          if (j < rows && lu.get(j, j) !== 0) {
            for (i = j + 1; i < rows; i++) {
              lu.set(i, j, lu.get(i, j) / lu.get(j, j));
            }
          }
        }
        this.LU = lu;
        this.pivotVector = pivotVector;
        this.pivotSign = pivotSign;
      }
      isSingular() {
        let data = this.LU;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          if (data.get(j, j) === 0) {
            return true;
          }
        }
        return false;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let lu = this.LU;
        let rows = lu.rows;
        if (rows !== value.rows) {
          throw new Error('Invalid matrix dimensions');
        }
        if (this.isSingular()) {
          throw new Error('LU matrix is singular');
        }
        let count = value.columns;
        let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
        let columns = lu.columns;
        let i, j, k;
        for (k = 0; k < columns; k++) {
          for (i = k + 1; i < columns; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        for (k = columns - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / lu.get(k, k));
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        return X;
      }
      get determinant() {
        let data = this.LU;
        if (!data.isSquare()) {
          throw new Error('Matrix must be square');
        }
        let determinant = this.pivotSign;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          determinant *= data.get(j, j);
        }
        return determinant;
      }
      get lowerTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i > j) {
              X.set(i, j, data.get(i, j));
            } else if (i === j) {
              X.set(i, j, 1);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get upperTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i <= j) {
              X.set(i, j, data.get(i, j));
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get pivotPermutationVector() {
        return Array.from(this.pivotVector);
      }
    }
    function hypotenuse(a, b) {
      let r = 0;
      if (Math.abs(a) > Math.abs(b)) {
        r = b / a;
        return Math.abs(a) * Math.sqrt(1 + r * r);
      }
      if (b !== 0) {
        r = a / b;
        return Math.abs(b) * Math.sqrt(1 + r * r);
      }
      return 0;
    }
    class QrDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let qr = value.clone();
        let m = value.rows;
        let n = value.columns;
        let rdiag = new Float64Array(n);
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          let nrm = 0;
          for (i = k; i < m; i++) {
            nrm = hypotenuse(nrm, qr.get(i, k));
          }
          if (nrm !== 0) {
            if (qr.get(k, k) < 0) {
              nrm = -nrm;
            }
            for (i = k; i < m; i++) {
              qr.set(i, k, qr.get(i, k) / nrm);
            }
            qr.set(k, k, qr.get(k, k) + 1);
            for (j = k + 1; j < n; j++) {
              s = 0;
              for (i = k; i < m; i++) {
                s += qr.get(i, k) * qr.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < m; i++) {
                qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
              }
            }
          }
          rdiag[k] = -nrm;
        }
        this.QR = qr;
        this.Rdiag = rdiag;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let qr = this.QR;
        let m = qr.rows;
        if (value.rows !== m) {
          throw new Error('Matrix row dimensions must agree');
        }
        if (!this.isFullRank()) {
          throw new Error('Matrix is rank deficient');
        }
        let count = value.columns;
        let X = value.clone();
        let n = qr.columns;
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          for (j = 0; j < count; j++) {
            s = 0;
            for (i = k; i < m; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }
            s = -s / qr.get(k, k);
            for (i = k; i < m; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
        for (k = n - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / this.Rdiag[k]);
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
            }
          }
        }
        return X.subMatrix(0, n - 1, 0, count - 1);
      }
      isFullRank() {
        let columns = this.QR.columns;
        for (let i = 0; i < columns; i++) {
          if (this.Rdiag[i] === 0) {
            return false;
          }
        }
        return true;
      }
      get upperTriangularMatrix() {
        let qr = this.QR;
        let n = qr.columns;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            if (i < j) {
              X.set(i, j, qr.get(i, j));
            } else if (i === j) {
              X.set(i, j, this.Rdiag[i]);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get orthogonalMatrix() {
        let qr = this.QR;
        let rows = qr.rows;
        let columns = qr.columns;
        let X = new Matrix$1(rows, columns);
        let i, j, k, s;
        for (k = columns - 1; k >= 0; k--) {
          for (i = 0; i < rows; i++) {
            X.set(i, k, 0);
          }
          X.set(k, k, 1);
          for (j = k; j < columns; j++) {
            if (qr.get(k, k) !== 0) {
              s = 0;
              for (i = k; i < rows; i++) {
                s += qr.get(i, k) * X.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < rows; i++) {
                X.set(i, j, X.get(i, j) + s * qr.get(i, k));
              }
            }
          }
        }
        return X;
      }
    }
    class SingularValueDecomposition {
      constructor(value, options = {}) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (value.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let m = value.rows;
        let n = value.columns;
        const {
          computeLeftSingularVectors = true,
          computeRightSingularVectors = true,
          autoTranspose = false
        } = options;
        let wantu = Boolean(computeLeftSingularVectors);
        let wantv = Boolean(computeRightSingularVectors);
        let swapped = false;
        let a;
        if (m < n) {
          if (!autoTranspose) {
            a = value.clone();
            // eslint-disable-next-line no-console
            console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
          } else {
            a = value.transpose();
            m = a.rows;
            n = a.columns;
            swapped = true;
            let aux = wantu;
            wantu = wantv;
            wantv = aux;
          }
        } else {
          a = value.clone();
        }
        let nu = Math.min(m, n);
        let ni = Math.min(m + 1, n);
        let s = new Float64Array(ni);
        let U = new Matrix$1(m, nu);
        let V = new Matrix$1(n, n);
        let e = new Float64Array(n);
        let work = new Float64Array(m);
        let si = new Float64Array(ni);
        for (let i = 0; i < ni; i++) si[i] = i;
        let nct = Math.min(m - 1, n);
        let nrt = Math.max(0, Math.min(n - 2, m));
        let mrc = Math.max(nct, nrt);
        for (let k = 0; k < mrc; k++) {
          if (k < nct) {
            s[k] = 0;
            for (let i = k; i < m; i++) {
              s[k] = hypotenuse(s[k], a.get(i, k));
            }
            if (s[k] !== 0) {
              if (a.get(k, k) < 0) {
                s[k] = -s[k];
              }
              for (let i = k; i < m; i++) {
                a.set(i, k, a.get(i, k) / s[k]);
              }
              a.set(k, k, a.get(k, k) + 1);
            }
            s[k] = -s[k];
          }
          for (let j = k + 1; j < n; j++) {
            if (k < nct && s[k] !== 0) {
              let t = 0;
              for (let i = k; i < m; i++) {
                t += a.get(i, k) * a.get(i, j);
              }
              t = -t / a.get(k, k);
              for (let i = k; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * a.get(i, k));
              }
            }
            e[j] = a.get(k, j);
          }
          if (wantu && k < nct) {
            for (let i = k; i < m; i++) {
              U.set(i, k, a.get(i, k));
            }
          }
          if (k < nrt) {
            e[k] = 0;
            for (let i = k + 1; i < n; i++) {
              e[k] = hypotenuse(e[k], e[i]);
            }
            if (e[k] !== 0) {
              if (e[k + 1] < 0) {
                e[k] = 0 - e[k];
              }
              for (let i = k + 1; i < n; i++) {
                e[i] /= e[k];
              }
              e[k + 1] += 1;
            }
            e[k] = -e[k];
            if (k + 1 < m && e[k] !== 0) {
              for (let i = k + 1; i < m; i++) {
                work[i] = 0;
              }
              for (let i = k + 1; i < m; i++) {
                for (let j = k + 1; j < n; j++) {
                  work[i] += e[j] * a.get(i, j);
                }
              }
              for (let j = k + 1; j < n; j++) {
                let t = -e[j] / e[k + 1];
                for (let i = k + 1; i < m; i++) {
                  a.set(i, j, a.get(i, j) + t * work[i]);
                }
              }
            }
            if (wantv) {
              for (let i = k + 1; i < n; i++) {
                V.set(i, k, e[i]);
              }
            }
          }
        }
        let p = Math.min(n, m + 1);
        if (nct < n) {
          s[nct] = a.get(nct, nct);
        }
        if (m < p) {
          s[p - 1] = 0;
        }
        if (nrt + 1 < p) {
          e[nrt] = a.get(nrt, p - 1);
        }
        e[p - 1] = 0;
        if (wantu) {
          for (let j = nct; j < nu; j++) {
            for (let i = 0; i < m; i++) {
              U.set(i, j, 0);
            }
            U.set(j, j, 1);
          }
          for (let k = nct - 1; k >= 0; k--) {
            if (s[k] !== 0) {
              for (let j = k + 1; j < nu; j++) {
                let t = 0;
                for (let i = k; i < m; i++) {
                  t += U.get(i, k) * U.get(i, j);
                }
                t = -t / U.get(k, k);
                for (let i = k; i < m; i++) {
                  U.set(i, j, U.get(i, j) + t * U.get(i, k));
                }
              }
              for (let i = k; i < m; i++) {
                U.set(i, k, -U.get(i, k));
              }
              U.set(k, k, 1 + U.get(k, k));
              for (let i = 0; i < k - 1; i++) {
                U.set(i, k, 0);
              }
            } else {
              for (let i = 0; i < m; i++) {
                U.set(i, k, 0);
              }
              U.set(k, k, 1);
            }
          }
        }
        if (wantv) {
          for (let k = n - 1; k >= 0; k--) {
            if (k < nrt && e[k] !== 0) {
              for (let j = k + 1; j < n; j++) {
                let t = 0;
                for (let i = k + 1; i < n; i++) {
                  t += V.get(i, k) * V.get(i, j);
                }
                t = -t / V.get(k + 1, k);
                for (let i = k + 1; i < n; i++) {
                  V.set(i, j, V.get(i, j) + t * V.get(i, k));
                }
              }
            }
            for (let i = 0; i < n; i++) {
              V.set(i, k, 0);
            }
            V.set(k, k, 1);
          }
        }
        let pp = p - 1;
        let eps = Number.EPSILON;
        while (p > 0) {
          let k, kase;
          for (k = p - 2; k >= -1; k--) {
            if (k === -1) {
              break;
            }
            const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
            if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
              e[k] = 0;
              break;
            }
          }
          if (k === p - 2) {
            kase = 4;
          } else {
            let ks;
            for (ks = p - 1; ks >= k; ks--) {
              if (ks === k) {
                break;
              }
              let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
              if (Math.abs(s[ks]) <= eps * t) {
                s[ks] = 0;
                break;
              }
            }
            if (ks === k) {
              kase = 3;
            } else if (ks === p - 1) {
              kase = 1;
            } else {
              kase = 2;
              k = ks;
            }
          }
          k++;
          switch (kase) {
            case 1:
              {
                let f = e[p - 2];
                e[p - 2] = 0;
                for (let j = p - 2; j >= k; j--) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  if (j !== k) {
                    f = -sn * e[j - 1];
                    e[j - 1] = cs * e[j - 1];
                  }
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                      V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                      V.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 2:
              {
                let f = e[k - 1];
                e[k - 1] = 0;
                for (let j = k; j < p; j++) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  f = -sn * e[j];
                  e[j] = cs * e[j];
                  if (wantu) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                      U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                      U.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 3:
              {
                const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
                const sp = s[p - 1] / scale;
                const spm1 = s[p - 2] / scale;
                const epm1 = e[p - 2] / scale;
                const sk = s[k] / scale;
                const ek = e[k] / scale;
                const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
                const c = sp * epm1 * (sp * epm1);
                let shift = 0;
                if (b !== 0 || c !== 0) {
                  if (b < 0) {
                    shift = 0 - Math.sqrt(b * b + c);
                  } else {
                    shift = Math.sqrt(b * b + c);
                  }
                  shift = c / (b + shift);
                }
                let f = (sk + sp) * (sk - sp) + shift;
                let g = sk * ek;
                for (let j = k; j < p - 1; j++) {
                  let t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  let cs = f / t;
                  let sn = g / t;
                  if (j !== k) {
                    e[j - 1] = t;
                  }
                  f = cs * s[j] + sn * e[j];
                  e[j] = cs * e[j] - sn * s[j];
                  g = sn * s[j + 1];
                  s[j + 1] = cs * s[j + 1];
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                      V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                      V.set(i, j, t);
                    }
                  }
                  t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  cs = f / t;
                  sn = g / t;
                  s[j] = t;
                  f = cs * e[j] + sn * s[j + 1];
                  s[j + 1] = -sn * e[j] + cs * s[j + 1];
                  g = sn * e[j + 1];
                  e[j + 1] = cs * e[j + 1];
                  if (wantu && j < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                      U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                      U.set(i, j, t);
                    }
                  }
                }
                e[p - 2] = f;
                break;
              }
            case 4:
              {
                if (s[k] <= 0) {
                  s[k] = s[k] < 0 ? -s[k] : 0;
                  if (wantv) {
                    for (let i = 0; i <= pp; i++) {
                      V.set(i, k, -V.get(i, k));
                    }
                  }
                }
                while (k < pp) {
                  if (s[k] >= s[k + 1]) {
                    break;
                  }
                  let t = s[k];
                  s[k] = s[k + 1];
                  s[k + 1] = t;
                  if (wantv && k < n - 1) {
                    for (let i = 0; i < n; i++) {
                      t = V.get(i, k + 1);
                      V.set(i, k + 1, V.get(i, k));
                      V.set(i, k, t);
                    }
                  }
                  if (wantu && k < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = U.get(i, k + 1);
                      U.set(i, k + 1, U.get(i, k));
                      U.set(i, k, t);
                    }
                  }
                  k++;
                }
                p--;
                break;
              }
            // no default
          }
        }
        if (swapped) {
          let tmp = V;
          V = U;
          U = tmp;
        }
        this.m = m;
        this.n = n;
        this.s = s;
        this.U = U;
        this.V = V;
      }
      solve(value) {
        let Y = value;
        let e = this.threshold;
        let scols = this.s.length;
        let Ls = Matrix$1.zeros(scols, scols);
        for (let i = 0; i < scols; i++) {
          if (Math.abs(this.s[i]) <= e) {
            Ls.set(i, i, 0);
          } else {
            Ls.set(i, i, 1 / this.s[i]);
          }
        }
        let U = this.U;
        let V = this.rightSingularVectors;
        let VL = V.mmul(Ls);
        let vrows = V.rows;
        let urows = U.rows;
        let VLU = Matrix$1.zeros(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < scols; k++) {
              sum += VL.get(i, k) * U.get(j, k);
            }
            VLU.set(i, j, sum);
          }
        }
        return VLU.mmul(Y);
      }
      solveForDiagonal(value) {
        return this.solve(Matrix$1.diag(value));
      }
      inverse() {
        let V = this.V;
        let e = this.threshold;
        let vrows = V.rows;
        let vcols = V.columns;
        let X = new Matrix$1(vrows, this.s.length);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < vcols; j++) {
            if (Math.abs(this.s[j]) > e) {
              X.set(i, j, V.get(i, j) / this.s[j]);
            }
          }
        }
        let U = this.U;
        let urows = U.rows;
        let ucols = U.columns;
        let Y = new Matrix$1(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < ucols; k++) {
              sum += X.get(i, k) * U.get(j, k);
            }
            Y.set(i, j, sum);
          }
        }
        return Y;
      }
      get condition() {
        return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
      }
      get norm2() {
        return this.s[0];
      }
      get rank() {
        let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
        let r = 0;
        let s = this.s;
        for (let i = 0, ii = s.length; i < ii; i++) {
          if (s[i] > tol) {
            r++;
          }
        }
        return r;
      }
      get diagonal() {
        return Array.from(this.s);
      }
      get threshold() {
        return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
      }
      get leftSingularVectors() {
        return this.U;
      }
      get rightSingularVectors() {
        return this.V;
      }
      get diagonalMatrix() {
        return Matrix$1.diag(this.s);
      }
    }
    function inverse(matrix, useSVD = false) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      if (useSVD) {
        return new SingularValueDecomposition(matrix).inverse();
      } else {
        return solve(matrix, Matrix$1.eye(matrix.rows));
      }
    }
    function solve(leftHandSide, rightHandSide, useSVD = false) {
      leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
      rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
      if (useSVD) {
        return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
      } else {
        return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
      }
    }
    function determinant(matrix) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isSquare()) {
        if (matrix.columns === 0) {
          return 1;
        }
        let a, b, c, d;
        if (matrix.columns === 2) {
          // 2 x 2 matrix
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(1, 0);
          d = matrix.get(1, 1);
          return a * d - b * c;
        } else if (matrix.columns === 3) {
          // 3 x 3 matrix
          let subMatrix0, subMatrix1, subMatrix2;
          subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
          subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
          subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(0, 2);
          return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
        } else {
          // general purpose determinant using the LU decomposition
          return new LuDecomposition(matrix).determinant;
        }
      } else {
        throw Error('determinant can only be calculated for a square matrix');
      }
    }
    function xrange(n, exception) {
      let range = [];
      for (let i = 0; i < n; i++) {
        if (i !== exception) {
          range.push(i);
        }
      }
      return range;
    }
    function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
      if (error > thresholdError) {
        return new Array(matrix.rows + 1).fill(0);
      } else {
        let returnArray = matrix.addRow(index, [0]);
        for (let i = 0; i < returnArray.rows; i++) {
          if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
            returnArray.set(i, 0, 0);
          }
        }
        return returnArray.to1DArray();
      }
    }
    function linearDependencies(matrix, options = {}) {
      const {
        thresholdValue = 10e-10,
        thresholdError = 10e-10
      } = options;
      matrix = Matrix$1.checkMatrix(matrix);
      let n = matrix.rows;
      let results = new Matrix$1(n, n);
      for (let i = 0; i < n; i++) {
        let b = Matrix$1.columnVector(matrix.getRow(i));
        let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
        let svd = new SingularValueDecomposition(Abis);
        let x = svd.solve(b);
        let error = Matrix$1.sub(b, Abis.mmul(x)).abs().max();
        results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
      }
      return results;
    }
    function pseudoInverse(matrix, threshold = Number.EPSILON) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isEmpty()) {
        // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
        // (0xn)*(nx0)*(0xn) = 0xn
        // (nx0)*(0xn)*(nx0) = nx0
        return matrix.transpose();
      }
      let svdSolution = new SingularValueDecomposition(matrix, {
        autoTranspose: true
      });
      let U = svdSolution.leftSingularVectors;
      let V = svdSolution.rightSingularVectors;
      let s = svdSolution.diagonal;
      for (let i = 0; i < s.length; i++) {
        if (Math.abs(s[i]) > threshold) {
          s[i] = 1.0 / s[i];
        } else {
          s[i] = 0.0;
        }
      }
      return V.mmul(Matrix$1.diag(s).mmul(U.transpose()));
    }
    function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true
      } = options;
      if (center) {
        xMatrix = xMatrix.center('column');
        if (!yIsSame) {
          yMatrix = yMatrix.center('column');
        }
      }
      const cov = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < cov.rows; i++) {
        for (let j = 0; j < cov.columns; j++) {
          cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
        }
      }
      return cov;
    }
    function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true,
        scale = true
      } = options;
      if (center) {
        xMatrix.center('column');
        if (!yIsSame) {
          yMatrix.center('column');
        }
      }
      if (scale) {
        xMatrix.scale('column');
        if (!yIsSame) {
          yMatrix.scale('column');
        }
      }
      const sdx = xMatrix.standardDeviation('column', {
        unbiased: true
      });
      const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
        unbiased: true
      });
      const corr = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < corr.rows; i++) {
        for (let j = 0; j < corr.columns; j++) {
          corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
        }
      }
      return corr;
    }
    class EigenvalueDecomposition {
      constructor(matrix, options = {}) {
        const {
          assumeSymmetric = false
        } = options;
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        if (!matrix.isSquare()) {
          throw new Error('Matrix is not a square matrix');
        }
        if (matrix.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let n = matrix.columns;
        let V = new Matrix$1(n, n);
        let d = new Float64Array(n);
        let e = new Float64Array(n);
        let value = matrix;
        let i, j;
        let isSymmetric = false;
        if (assumeSymmetric) {
          isSymmetric = true;
        } else {
          isSymmetric = matrix.isSymmetric();
        }
        if (isSymmetric) {
          for (i = 0; i < n; i++) {
            for (j = 0; j < n; j++) {
              V.set(i, j, value.get(i, j));
            }
          }
          tred2(n, e, d, V);
          tql2(n, e, d, V);
        } else {
          let H = new Matrix$1(n, n);
          let ort = new Float64Array(n);
          for (j = 0; j < n; j++) {
            for (i = 0; i < n; i++) {
              H.set(i, j, value.get(i, j));
            }
          }
          orthes(n, H, ort, V);
          hqr2(n, e, d, V, H);
        }
        this.n = n;
        this.e = e;
        this.d = d;
        this.V = V;
      }
      get realEigenvalues() {
        return Array.from(this.d);
      }
      get imaginaryEigenvalues() {
        return Array.from(this.e);
      }
      get eigenvectorMatrix() {
        return this.V;
      }
      get diagonalMatrix() {
        let n = this.n;
        let e = this.e;
        let d = this.d;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            X.set(i, j, 0);
          }
          X.set(i, i, d[i]);
          if (e[i] > 0) {
            X.set(i, i + 1, e[i]);
          } else if (e[i] < 0) {
            X.set(i, i - 1, e[i]);
          }
        }
        return X;
      }
    }
    function tred2(n, e, d, V) {
      let f, g, h, i, j, k, hh, scale;
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
      }
      for (i = n - 1; i > 0; i--) {
        scale = 0;
        h = 0;
        for (k = 0; k < i; k++) {
          scale = scale + Math.abs(d[k]);
        }
        if (scale === 0) {
          e[i] = d[i - 1];
          for (j = 0; j < i; j++) {
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
            V.set(j, i, 0);
          }
        } else {
          for (k = 0; k < i; k++) {
            d[k] /= scale;
            h += d[k] * d[k];
          }
          f = d[i - 1];
          g = Math.sqrt(h);
          if (f > 0) {
            g = -g;
          }
          e[i] = scale * g;
          h = h - f * g;
          d[i - 1] = f - g;
          for (j = 0; j < i; j++) {
            e[j] = 0;
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            V.set(j, i, f);
            g = e[j] + V.get(j, j) * f;
            for (k = j + 1; k <= i - 1; k++) {
              g += V.get(k, j) * d[k];
              e[k] += V.get(k, j) * f;
            }
            e[j] = g;
          }
          f = 0;
          for (j = 0; j < i; j++) {
            e[j] /= h;
            f += e[j] * d[j];
          }
          hh = f / (h + h);
          for (j = 0; j < i; j++) {
            e[j] -= hh * d[j];
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            g = e[j];
            for (k = j; k <= i - 1; k++) {
              V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
            }
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
          }
        }
        d[i] = h;
      }
      for (i = 0; i < n - 1; i++) {
        V.set(n - 1, i, V.get(i, i));
        V.set(i, i, 1);
        h = d[i + 1];
        if (h !== 0) {
          for (k = 0; k <= i; k++) {
            d[k] = V.get(k, i + 1) / h;
          }
          for (j = 0; j <= i; j++) {
            g = 0;
            for (k = 0; k <= i; k++) {
              g += V.get(k, i + 1) * V.get(k, j);
            }
            for (k = 0; k <= i; k++) {
              V.set(k, j, V.get(k, j) - g * d[k]);
            }
          }
        }
        for (k = 0; k <= i; k++) {
          V.set(k, i + 1, 0);
        }
      }
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
        V.set(n - 1, j, 0);
      }
      V.set(n - 1, n - 1, 1);
      e[0] = 0;
    }
    function tql2(n, e, d, V) {
      let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
      for (i = 1; i < n; i++) {
        e[i - 1] = e[i];
      }
      e[n - 1] = 0;
      let f = 0;
      let tst1 = 0;
      let eps = Number.EPSILON;
      for (l = 0; l < n; l++) {
        tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
        m = l;
        while (m < n) {
          if (Math.abs(e[m]) <= eps * tst1) {
            break;
          }
          m++;
        }
        if (m > l) {
          do {
            g = d[l];
            p = (d[l + 1] - g) / (2 * e[l]);
            r = hypotenuse(p, 1);
            if (p < 0) {
              r = -r;
            }
            d[l] = e[l] / (p + r);
            d[l + 1] = e[l] * (p + r);
            dl1 = d[l + 1];
            h = g - d[l];
            for (i = l + 2; i < n; i++) {
              d[i] -= h;
            }
            f = f + h;
            p = d[m];
            c = 1;
            c2 = c;
            c3 = c;
            el1 = e[l + 1];
            s = 0;
            s2 = 0;
            for (i = m - 1; i >= l; i--) {
              c3 = c2;
              c2 = c;
              s2 = s;
              g = c * e[i];
              h = c * p;
              r = hypotenuse(p, e[i]);
              e[i + 1] = s * r;
              s = e[i] / r;
              c = p / r;
              p = c * d[i] - s * g;
              d[i + 1] = h + s * (c * g + s * d[i]);
              for (k = 0; k < n; k++) {
                h = V.get(k, i + 1);
                V.set(k, i + 1, s * V.get(k, i) + c * h);
                V.set(k, i, c * V.get(k, i) - s * h);
              }
            }
            p = -s * s2 * c3 * el1 * e[l] / dl1;
            e[l] = s * p;
            d[l] = c * p;
          } while (Math.abs(e[l]) > eps * tst1);
        }
        d[l] = d[l] + f;
        e[l] = 0;
      }
      for (i = 0; i < n - 1; i++) {
        k = i;
        p = d[i];
        for (j = i + 1; j < n; j++) {
          if (d[j] < p) {
            k = j;
            p = d[j];
          }
        }
        if (k !== i) {
          d[k] = d[i];
          d[i] = p;
          for (j = 0; j < n; j++) {
            p = V.get(j, i);
            V.set(j, i, V.get(j, k));
            V.set(j, k, p);
          }
        }
      }
    }
    function orthes(n, H, ort, V) {
      let low = 0;
      let high = n - 1;
      let f, g, h, i, j, m;
      let scale;
      for (m = low + 1; m <= high - 1; m++) {
        scale = 0;
        for (i = m; i <= high; i++) {
          scale = scale + Math.abs(H.get(i, m - 1));
        }
        if (scale !== 0) {
          h = 0;
          for (i = high; i >= m; i--) {
            ort[i] = H.get(i, m - 1) / scale;
            h += ort[i] * ort[i];
          }
          g = Math.sqrt(h);
          if (ort[m] > 0) {
            g = -g;
          }
          h = h - ort[m] * g;
          ort[m] = ort[m] - g;
          for (j = m; j < n; j++) {
            f = 0;
            for (i = high; i >= m; i--) {
              f += ort[i] * H.get(i, j);
            }
            f = f / h;
            for (i = m; i <= high; i++) {
              H.set(i, j, H.get(i, j) - f * ort[i]);
            }
          }
          for (i = 0; i <= high; i++) {
            f = 0;
            for (j = high; j >= m; j--) {
              f += ort[j] * H.get(i, j);
            }
            f = f / h;
            for (j = m; j <= high; j++) {
              H.set(i, j, H.get(i, j) - f * ort[j]);
            }
          }
          ort[m] = scale * ort[m];
          H.set(m, m - 1, scale * g);
        }
      }
      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          V.set(i, j, i === j ? 1 : 0);
        }
      }
      for (m = high - 1; m >= low + 1; m--) {
        if (H.get(m, m - 1) !== 0) {
          for (i = m + 1; i <= high; i++) {
            ort[i] = H.get(i, m - 1);
          }
          for (j = m; j <= high; j++) {
            g = 0;
            for (i = m; i <= high; i++) {
              g += ort[i] * V.get(i, j);
            }
            g = g / ort[m] / H.get(m, m - 1);
            for (i = m; i <= high; i++) {
              V.set(i, j, V.get(i, j) + g * ort[i]);
            }
          }
        }
      }
    }
    function hqr2(nn, e, d, V, H) {
      let n = nn - 1;
      let low = 0;
      let high = nn - 1;
      let eps = Number.EPSILON;
      let exshift = 0;
      let norm = 0;
      let p = 0;
      let q = 0;
      let r = 0;
      let s = 0;
      let z = 0;
      let iter = 0;
      let i, j, k, l, m, t, w, x, y;
      let ra, sa, vr, vi;
      let notlast, cdivres;
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          d[i] = H.get(i, i);
          e[i] = 0;
        }
        for (j = Math.max(i - 1, 0); j < nn; j++) {
          norm = norm + Math.abs(H.get(i, j));
        }
      }
      while (n >= low) {
        l = n;
        while (l > low) {
          s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
          if (s === 0) {
            s = norm;
          }
          if (Math.abs(H.get(l, l - 1)) < eps * s) {
            break;
          }
          l--;
        }
        if (l === n) {
          H.set(n, n, H.get(n, n) + exshift);
          d[n] = H.get(n, n);
          e[n] = 0;
          n--;
          iter = 0;
        } else if (l === n - 1) {
          w = H.get(n, n - 1) * H.get(n - 1, n);
          p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
          q = p * p + w;
          z = Math.sqrt(Math.abs(q));
          H.set(n, n, H.get(n, n) + exshift);
          H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
          x = H.get(n, n);
          if (q >= 0) {
            z = p >= 0 ? p + z : p - z;
            d[n - 1] = x + z;
            d[n] = d[n - 1];
            if (z !== 0) {
              d[n] = x - w / z;
            }
            e[n - 1] = 0;
            e[n] = 0;
            x = H.get(n, n - 1);
            s = Math.abs(x) + Math.abs(z);
            p = x / s;
            q = z / s;
            r = Math.sqrt(p * p + q * q);
            p = p / r;
            q = q / r;
            for (j = n - 1; j < nn; j++) {
              z = H.get(n - 1, j);
              H.set(n - 1, j, q * z + p * H.get(n, j));
              H.set(n, j, q * H.get(n, j) - p * z);
            }
            for (i = 0; i <= n; i++) {
              z = H.get(i, n - 1);
              H.set(i, n - 1, q * z + p * H.get(i, n));
              H.set(i, n, q * H.get(i, n) - p * z);
            }
            for (i = low; i <= high; i++) {
              z = V.get(i, n - 1);
              V.set(i, n - 1, q * z + p * V.get(i, n));
              V.set(i, n, q * V.get(i, n) - p * z);
            }
          } else {
            d[n - 1] = x + p;
            d[n] = x + p;
            e[n - 1] = z;
            e[n] = -z;
          }
          n = n - 2;
          iter = 0;
        } else {
          x = H.get(n, n);
          y = 0;
          w = 0;
          if (l < n) {
            y = H.get(n - 1, n - 1);
            w = H.get(n, n - 1) * H.get(n - 1, n);
          }
          if (iter === 10) {
            exshift += x;
            for (i = low; i <= n; i++) {
              H.set(i, i, H.get(i, i) - x);
            }
            s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
            // eslint-disable-next-line no-multi-assign
            x = y = 0.75 * s;
            w = -0.4375 * s * s;
          }
          if (iter === 30) {
            s = (y - x) / 2;
            s = s * s + w;
            if (s > 0) {
              s = Math.sqrt(s);
              if (y < x) {
                s = -s;
              }
              s = x - w / ((y - x) / 2 + s);
              for (i = low; i <= n; i++) {
                H.set(i, i, H.get(i, i) - s);
              }
              exshift += s;
              // eslint-disable-next-line no-multi-assign
              x = y = w = 0.964;
            }
          }
          iter = iter + 1;
          m = n - 2;
          while (m >= l) {
            z = H.get(m, m);
            r = x - z;
            s = y - z;
            p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
            q = H.get(m + 1, m + 1) - z - r - s;
            r = H.get(m + 2, m + 1);
            s = Math.abs(p) + Math.abs(q) + Math.abs(r);
            p = p / s;
            q = q / s;
            r = r / s;
            if (m === l) {
              break;
            }
            if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
              break;
            }
            m--;
          }
          for (i = m + 2; i <= n; i++) {
            H.set(i, i - 2, 0);
            if (i > m + 2) {
              H.set(i, i - 3, 0);
            }
          }
          for (k = m; k <= n - 1; k++) {
            notlast = k !== n - 1;
            if (k !== m) {
              p = H.get(k, k - 1);
              q = H.get(k + 1, k - 1);
              r = notlast ? H.get(k + 2, k - 1) : 0;
              x = Math.abs(p) + Math.abs(q) + Math.abs(r);
              if (x !== 0) {
                p = p / x;
                q = q / x;
                r = r / x;
              }
            }
            if (x === 0) {
              break;
            }
            s = Math.sqrt(p * p + q * q + r * r);
            if (p < 0) {
              s = -s;
            }
            if (s !== 0) {
              if (k !== m) {
                H.set(k, k - 1, -s * x);
              } else if (l !== m) {
                H.set(k, k - 1, -H.get(k, k - 1));
              }
              p = p + s;
              x = p / s;
              y = q / s;
              z = r / s;
              q = q / p;
              r = r / p;
              for (j = k; j < nn; j++) {
                p = H.get(k, j) + q * H.get(k + 1, j);
                if (notlast) {
                  p = p + r * H.get(k + 2, j);
                  H.set(k + 2, j, H.get(k + 2, j) - p * z);
                }
                H.set(k, j, H.get(k, j) - p * x);
                H.set(k + 1, j, H.get(k + 1, j) - p * y);
              }
              for (i = 0; i <= Math.min(n, k + 3); i++) {
                p = x * H.get(i, k) + y * H.get(i, k + 1);
                if (notlast) {
                  p = p + z * H.get(i, k + 2);
                  H.set(i, k + 2, H.get(i, k + 2) - p * r);
                }
                H.set(i, k, H.get(i, k) - p);
                H.set(i, k + 1, H.get(i, k + 1) - p * q);
              }
              for (i = low; i <= high; i++) {
                p = x * V.get(i, k) + y * V.get(i, k + 1);
                if (notlast) {
                  p = p + z * V.get(i, k + 2);
                  V.set(i, k + 2, V.get(i, k + 2) - p * r);
                }
                V.set(i, k, V.get(i, k) - p);
                V.set(i, k + 1, V.get(i, k + 1) - p * q);
              }
            }
          }
        }
      }
      if (norm === 0) {
        return;
      }
      for (n = nn - 1; n >= 0; n--) {
        p = d[n];
        q = e[n];
        if (q === 0) {
          l = n;
          H.set(n, n, 1);
          for (i = n - 1; i >= 0; i--) {
            w = H.get(i, i) - p;
            r = 0;
            for (j = l; j <= n; j++) {
              r = r + H.get(i, j) * H.get(j, n);
            }
            if (e[i] < 0) {
              z = w;
              s = r;
            } else {
              l = i;
              if (e[i] === 0) {
                H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
                t = (x * s - z * r) / q;
                H.set(i, n, t);
                H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
              }
              t = Math.abs(H.get(i, n));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        } else if (q < 0) {
          l = n - 1;
          if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
            H.set(n - 1, n - 1, q / H.get(n, n - 1));
            H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
          } else {
            cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
            H.set(n - 1, n - 1, cdivres[0]);
            H.set(n - 1, n, cdivres[1]);
          }
          H.set(n, n - 1, 0);
          H.set(n, n, 1);
          for (i = n - 2; i >= 0; i--) {
            ra = 0;
            sa = 0;
            for (j = l; j <= n; j++) {
              ra = ra + H.get(i, j) * H.get(j, n - 1);
              sa = sa + H.get(i, j) * H.get(j, n);
            }
            w = H.get(i, i) - p;
            if (e[i] < 0) {
              z = w;
              r = ra;
              s = sa;
            } else {
              l = i;
              if (e[i] === 0) {
                cdivres = cdiv(-ra, -sa, w, q);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
                vi = (d[i] - p) * 2 * q;
                if (vr === 0 && vi === 0) {
                  vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
                }
                cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
                if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
                  H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
                  H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
                } else {
                  cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
                  H.set(i + 1, n - 1, cdivres[0]);
                  H.set(i + 1, n, cdivres[1]);
                }
              }
              t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n - 1, H.get(j, n - 1) / t);
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        }
      }
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          for (j = i; j < nn; j++) {
            V.set(i, j, H.get(i, j));
          }
        }
      }
      for (j = nn - 1; j >= low; j--) {
        for (i = low; i <= high; i++) {
          z = 0;
          for (k = low; k <= Math.min(j, high); k++) {
            z = z + V.get(i, k) * H.get(k, j);
          }
          V.set(i, j, z);
        }
      }
    }
    function cdiv(xr, xi, yr, yi) {
      let r, d;
      if (Math.abs(yr) > Math.abs(yi)) {
        r = yi / yr;
        d = yr + r * yi;
        return [(xr + r * xi) / d, (xi - r * xr) / d];
      } else {
        r = yr / yi;
        d = yi + r * yr;
        return [(r * xr + xi) / d, (r * xi - xr) / d];
      }
    }
    class CholeskyDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (!value.isSymmetric()) {
          throw new Error('Matrix is not symmetric');
        }
        let a = value;
        let dimension = a.rows;
        let l = new Matrix$1(dimension, dimension);
        let positiveDefinite = true;
        let i, j, k;
        for (j = 0; j < dimension; j++) {
          let d = 0;
          for (k = 0; k < j; k++) {
            let s = 0;
            for (i = 0; i < k; i++) {
              s += l.get(k, i) * l.get(j, i);
            }
            s = (a.get(j, k) - s) / l.get(k, k);
            l.set(j, k, s);
            d = d + s * s;
          }
          d = a.get(j, j) - d;
          positiveDefinite &&= d > 0;
          l.set(j, j, Math.sqrt(Math.max(d, 0)));
          for (k = j + 1; k < dimension; k++) {
            l.set(j, k, 0);
          }
        }
        this.L = l;
        this.positiveDefinite = positiveDefinite;
      }
      isPositiveDefinite() {
        return this.positiveDefinite;
      }
      solve(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let l = this.L;
        let dimension = l.rows;
        if (value.rows !== dimension) {
          throw new Error('Matrix dimensions do not match');
        }
        if (this.isPositiveDefinite() === false) {
          throw new Error('Matrix is not positive definite');
        }
        let count = value.columns;
        let B = value.clone();
        let i, j, k;
        for (k = 0; k < dimension; k++) {
          for (j = 0; j < count; j++) {
            for (i = 0; i < k; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        for (k = dimension - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            for (i = k + 1; i < dimension; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        return B;
      }
      get lowerTriangularMatrix() {
        return this.L;
      }
    }
    class nipals {
      constructor(X, options = {}) {
        X = WrapperMatrix2D.checkMatrix(X);
        let {
          Y
        } = options;
        const {
          scaleScores = false,
          maxIterations = 1000,
          terminationCriteria = 1e-10
        } = options;
        let u;
        if (Y) {
          if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
            Y = Matrix$1.columnVector(Y);
          } else {
            Y = WrapperMatrix2D.checkMatrix(Y);
          }
          if (Y.rows !== X.rows) {
            throw new Error('Y should have the same number of rows as X');
          }
          u = Y.getColumnVector(0);
        } else {
          u = X.getColumnVector(0);
        }
        let diff = 1;
        let t, q, w, tOld;
        for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
          w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
          w = w.div(w.norm());
          t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
          if (counter > 0) {
            diff = t.clone().sub(tOld).pow(2).sum();
          }
          tOld = t.clone();
          if (Y) {
            q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
            q = q.div(q.norm());
            u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
          } else {
            u = t;
          }
        }
        if (Y) {
          let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          p = p.div(p.norm());
          let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
          let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
          this.t = t;
          this.p = p.transpose();
          this.w = w.transpose();
          this.q = q;
          this.u = u;
          this.s = t.transpose().mmul(t);
          this.xResidual = xResidual;
          this.yResidual = yResidual;
          this.betas = residual;
        } else {
          this.w = w.transpose();
          this.s = t.transpose().mmul(t).sqrt();
          if (scaleScores) {
            this.t = t.clone().div(this.s.get(0, 0));
          } else {
            this.t = t;
          }
          this.xResidual = X.sub(t.mmul(w.transpose()));
        }
      }
    }
    matrix.AbstractMatrix = AbstractMatrix;
    matrix.CHO = CholeskyDecomposition;
    matrix.CholeskyDecomposition = CholeskyDecomposition;
    matrix.DistanceMatrix = DistanceMatrix;
    matrix.EVD = EigenvalueDecomposition;
    matrix.EigenvalueDecomposition = EigenvalueDecomposition;
    matrix.LU = LuDecomposition;
    matrix.LuDecomposition = LuDecomposition;
    var Matrix_1 = matrix.Matrix = Matrix$1;
    matrix.MatrixColumnSelectionView = MatrixColumnSelectionView;
    matrix.MatrixColumnView = MatrixColumnView;
    matrix.MatrixFlipColumnView = MatrixFlipColumnView;
    matrix.MatrixFlipRowView = MatrixFlipRowView;
    matrix.MatrixRowSelectionView = MatrixRowSelectionView;
    matrix.MatrixRowView = MatrixRowView;
    matrix.MatrixSelectionView = MatrixSelectionView;
    matrix.MatrixSubView = MatrixSubView;
    matrix.MatrixTransposeView = MatrixTransposeView;
    matrix.NIPALS = nipals;
    matrix.Nipals = nipals;
    matrix.QR = QrDecomposition;
    matrix.QrDecomposition = QrDecomposition;
    matrix.SVD = SingularValueDecomposition;
    matrix.SingularValueDecomposition = SingularValueDecomposition;
    matrix.SymmetricMatrix = SymmetricMatrix;
    matrix.WrapperMatrix1D = WrapperMatrix1D;
    matrix.WrapperMatrix2D = WrapperMatrix2D;
    matrix.correlation = correlation;
    matrix.covariance = covariance;
    var _default = matrix.default = Matrix$1;
    matrix.determinant = determinant;
    matrix.inverse = inverse;
    matrix.linearDependencies = linearDependencies;
    matrix.pseudoInverse = pseudoInverse;
    matrix.solve = solve;
    matrix.wrap = wrap;

    const Matrix = Matrix_1;
    _default.Matrix ? _default.Matrix : Matrix_1;

    /**
     * Algorithm that finds the shortest distance from one node to the other
     * @param {Matrix} adjMatrix - A squared adjacency matrix
     * @return {Matrix} - Distance from a node to the other, -1 if the node is unreachable
     */
    function floydWarshall(adjMatrix) {
      if (Matrix.isMatrix(adjMatrix) && adjMatrix.columns !== adjMatrix.rows) {
        throw new TypeError('The adjacency matrix should be squared');
      }
      const numVertices = adjMatrix.columns;
      let distMatrix = new Matrix(numVertices, numVertices);
      distMatrix.apply((row, column) => {
        // principal diagonal is 0
        if (row === column) {
          distMatrix.set(row, column, 0);
        } else {
          let val = adjMatrix.get(row, column);
          if (val || Object.is(val, -0)) {
            // edges values remain the same
            distMatrix.set(row, column, val);
          } else {
            // 0 values become infinity
            distMatrix.set(row, column, Number.POSITIVE_INFINITY);
          }
        }
      });
      for (let k = 0; k < numVertices; ++k) {
        for (let i = 0; i < numVertices; ++i) {
          for (let j = 0; j < numVertices; ++j) {
            let dist = distMatrix.get(i, k) + distMatrix.get(k, j);
            if (distMatrix.get(i, j) > dist) {
              distMatrix.set(i, j, dist);
            }
          }
        }
      }
      // When there's no connection the value is -1
      distMatrix.apply((row, column) => {
        if (distMatrix.get(row, column) === Number.POSITIVE_INFINITY) {
          distMatrix.set(row, column, -1);
        }
      });
      return distMatrix;
    }

    /**
     * Returns a connectivity matrix
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {boolean} [options.pathLength=false] - get the path length between atoms
     * @param {boolean} [options.mass=false] - set the nominal mass of the atoms on diagonal
     * @param {boolean} [options.atomicNo=false] - set the atomic number of the atom on diagonal
     * @param {boolean} [options.negativeAtomicNo=false] - set the atomic number * -1 of the atom on diagonal
     * @param {boolean} [options.sdt=false] - set 1, 2 or 3 depending if single, double or triple bond
     * @param {boolean} [options.sdta=false] - set 1, 2, 3 or 4 depending if single, double, triple or aromatic  bond
     */
    function getConnectivityMatrix(molecule, options = {}) {
      const OCL = molecule.getOCL();
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      const nbAtoms = molecule.getAllAtoms();
      let result = new Array(nbAtoms).fill();
      result = result.map(() => new Array(nbAtoms).fill(0));
      if (!options.pathLength) {
        if (options.atomicNo) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = molecule.getAtomicNo(i);
          }
        } else if (options.negativeAtomicNo) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = -molecule.getAtomicNo(i);
          }
        } else if (options.mass) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = OCL.Molecule.cRoundedMass[molecule.getAtomicNo(i)];
          }
        } else {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = 1;
          }
        }
      }
      if (options.sdt) {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
          }
        }
      } else if (options.sdta) {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            const bondNumber = molecule.getConnBond(i, j);
            if (molecule.isAromaticBond(bondNumber)) {
              result[i][molecule.getConnAtom(i, j)] = 4;
            } else {
              result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
            }
          }
        }
      } else {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            result[i][molecule.getConnAtom(i, j)] = 1;
          }
        }
      }
      if (options.pathLength) {
        result = floydWarshall(new Matrix(result)).to2DArray();
      }
      return result;
    }

    /**
     * Creates a compact copy of the molecule without custom labels.
     * We don't want that custom labels interfere with hose code and diaID generation.
     * @param molecule
     * @returns
     */
    function getCompactCopyWithoutCustomLabels(molecule) {
      const tempMolecule = molecule.getCompactCopy();
      for (let i = 0; i < tempMolecule.getAllAtoms(); i++) {
        tempMolecule.setAtomCustomLabel(i, '');
      }
      return tempMolecule;
    }

    function getCanonizedDiaIDs(diaMol, options) {
      const {
        logger,
        maxNbAtoms
      } = options;
      const moleculeWithH = diaMol.moleculeWithH;
      if (moleculeWithH.getAllAtoms() > maxNbAtoms) {
        logger.warn(`too many atoms to evaluate heterotopic chiral bonds: ${moleculeWithH.getAllAtoms()} > ${maxNbAtoms}`);
        return [];
      }
      const heterotopicSymmetryRanks = diaMol.heterotopicSymmetryRanks;
      const finalRanks = diaMol.finalRanks;
      const canonizedDiaIDs = new Array(moleculeWithH.getAllAtoms());
      moleculeWithH.ensureHelperArrays(diaMol.molecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const cache = {};
      for (let i = 0; i < diaMol.moleculeWithH.getAllAtoms(); i++) {
        const rank = heterotopicSymmetryRanks[i];
        if (rank && cache[rank]) {
          canonizedDiaIDs[finalRanks[i]] = cache[rank].diaID;
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(diaMol.moleculeWithH);
        tagAtom(tempMolecule, i);
        makeRacemic(tempMolecule);
        const diaID = tempMolecule.getCanonizedIDCode(diaMol.molecule.getOCL().Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        canonizedDiaIDs[finalRanks[i]] = diaID;
      }
      return canonizedDiaIDs;
    }

    /**
     * Returns an array of strings (idCodes) specified molecule. Each string corresponds to a
     * hose code. By default it will calculate the hose codes for sphere 0 to 4 and will reuse
     * the existing tagged atoms.
     * @param molecule - The OCL molecule to process.
     * @param options - Options for generating hose codes.
     * @returns An array of hose code strings.
     */
    function getHoseCodesForAtomsAsStrings(molecule, options = {}) {
      const fragments = getHoseCodesForAtomsAsFragments(molecule, options);
      const OCL = molecule.getOCL();
      const hoses = [];
      for (const fragment of fragments) {
        hoses.push(fragment.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS));
      }
      return hoses;
    }

    /**
     * Get the canonized hose codes for a topic molecule. It will use the moleculeWithH
     * @param topicMolecule - The topic molecule to get the hose codes for.
     * @returns The canonized hose codes.
     */
    function getCanonizedHoseCodes(topicMolecule) {
      const options = topicMolecule.options;
      const heterotopicSymmetryRanks = topicMolecule.heterotopicSymmetryRanks;
      const moleculeWithH = topicMolecule.moleculeWithH;
      const finalRanks = topicMolecule.finalRanks;
      const canonizedHoseCodes = new Array(moleculeWithH.getAllAtoms());
      moleculeWithH.ensureHelperArrays(topicMolecule.molecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const cache = {};
      for (let i = 0; i < topicMolecule.moleculeWithH.getAllAtoms(); i++) {
        const rank = heterotopicSymmetryRanks[i];
        if (rank && cache[rank]) {
          canonizedHoseCodes[finalRanks[i]] = cache[rank].diaID;
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(moleculeWithH);
        tagAtom(tempMolecule, i);
        const hoses = getHoseCodesForAtomsAsStrings(tempMolecule, options);
        canonizedHoseCodes[finalRanks[i]] = hoses;
      }
      return canonizedHoseCodes;
    }

    /**
     * For each atom we will return an array of objects
     * @param topicMolecule
     * @param options
     * @returns
     */
    function getCanonizedHoseCodesForPath(topicMolecule, options = {}) {
      const {
        minPathLength = 0,
        maxPathLength = topicMolecule.options.maxPathLength,
        toAtomicNo,
        fromAtomicNo
      } = options;
      if (maxPathLength > topicMolecule.options.maxPathLength) {
        throw new Error(`maxPathLength cannot be larger than the one defined in topicMolecule: ${topicMolecule.options.maxPathLength}`);
      }
      const atomsPaths = topicMolecule.atomsPaths;
      const molecule = topicMolecule.moleculeWithH;
      const results = new Array(molecule.getAllAtoms());
      for (let fromAtom = 0; fromAtom < molecule.getAllAtoms(); fromAtom++) {
        results[fromAtom] = {
          fromDiaID: topicMolecule.diaIDs[fromAtom],
          paths: []
        };
        if (fromAtomicNo && molecule.getAtomicNo(fromAtom) !== fromAtomicNo) {
          continue;
        }
        for (let pathLength = minPathLength; pathLength <= maxPathLength; pathLength++) {
          const pathOfSpecificLength = atomsPaths[fromAtom][pathLength];
          for (const path of pathOfSpecificLength) {
            const toAtom = path.path.at(-1);
            if (toAtomicNo && molecule.getAtomicNo(toAtom) !== toAtomicNo) {
              continue;
            }
            results[fromAtom].paths.push({
              toDiaID: topicMolecule.diaIDs[toAtom],
              pathLength: path.pathLength,
              path: path.path,
              hoses: getHoseCodesForAtomsAsStrings(topicMolecule.moleculeWithH, {
                ...options,
                rootAtoms: path.path,
                tagAtoms: [fromAtom, toAtom]
              })
            });
          }
        }
      }
      return results;
    }

    function getDiaIDsAndInfo(diaMol, canonizedDiaIDs) {
      const newDiaIDs = [];
      const molecule = diaMol.moleculeWithH;
      const counts = {};
      for (const diaID of canonizedDiaIDs) {
        if (!counts[diaID]) {
          counts[diaID] = 0;
        }
        counts[diaID]++;
      }
      for (let i = 0; i < canonizedDiaIDs.length; i++) {
        const diaID = canonizedDiaIDs[diaMol.finalRanks[i]];
        if (!diaID) {
          throw new Error(`Unexpected missing canonized diaID for atom ${i}`);
        }
        const count = counts[diaID];
        if (!count) {
          throw new Error(`Unexpected missing count for diaID ${diaID}`);
        }
        const newDiaID = {
          idCode: diaID,
          attachedHydrogensIDCodes: [],
          attachedHydrogens: [],
          nbAttachedHydrogens: 0,
          atomLabel: molecule.getAtomLabel(i),
          nbEquivalentAtoms: count,
          heavyAtom: undefined,
          atomMapNo: molecule.getAtomMapNo(i)
        };
        if (molecule.getAtomicNo(i) === 1) {
          const atom = molecule.getConnAtom(i, 0);
          newDiaID.heavyAtom = canonizedDiaIDs[diaMol.finalRanks[atom]];
        }
        for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
          const atom = molecule.getConnAtom(i, j);
          if (molecule.getAtomicNo(atom) === 1) {
            newDiaID.nbAttachedHydrogens++;
            newDiaID.attachedHydrogens.push(atom);
            const hydrogenDiaID = canonizedDiaIDs[diaMol.finalRanks[atom]];
            if (!hydrogenDiaID) {
              throw new Error(`Unexpected missing canonized diaID for atom ${atom}`);
            }
            if (!newDiaID.attachedHydrogensIDCodes.includes(hydrogenDiaID)) {
              newDiaID.attachedHydrogensIDCodes.push(hydrogenDiaID);
            }
          }
        }
        newDiaIDs.push(newDiaID);
      }
      return newDiaIDs;
    }

    /**
     * Get a unique atomic number for a X
     * @param xMolecule
     * @returns
     */
    function getHeterotopicSymmetryRanks(xMolecule) {
      xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const symmetryRanks = [];
      for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
        symmetryRanks.push(xMolecule.getSymmetryRank(i));
      }
      return symmetryRanks;
    }
    function getFinalRanks(xMolecule) {
      xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      return xMolecule.getFinalRanks(0).map(rank => rank - 1);
    }

    /**
     * Returns the atoms that are chiral or pseudo chiral.
     * There could be some issues if the original molecule lacks chiral bonds.
     * The function will add them and this could lead to some issues in the case of pseudochiral atoms.
     * @param {import('openchemlib').Molecule} molecule
     * @returns {number[]}
     */
    function getChiralOrHeterotopicCarbons(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      const xAtomicNumber = getXAtomicNumber(molecule);
      const internalMolecule = molecule.getCompactCopy();
      // hydrogens may be diastereotopic, we need to add them
      internalMolecule.addImplicitHydrogens();
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (internalMolecule.getAtomicNo(i) === 1) {
          internalMolecule.setAtomicNo(i, xAtomicNumber);
        }
      }
      addPossibleChiralBonds(internalMolecule);
      internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const atoms = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (internalMolecule.getAtomicNo(i) === xAtomicNumber) {
          continue;
        }
        if (molecule.getAtomicNo(i) !== internalMolecule.getAtomicNo(i)) {
          throw new Error('getChiralOrHeterotopicCarbons: mismatching atomic numbers');
        }
        if (internalMolecule.getAtomicNo(i) !== 6) {
          continue;
        }
        const neighbourSymmetries = getNeighbourSymmetries(internalMolecule, i);
        if (neighbourSymmetries.length === 4) {
          atoms.push(i);
        }
      }
      return atoms;
    }
    function addPossibleChiralBonds(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      for (let i = 0; i < molecule.getAtoms(); i++) {
        if (molecule.getAtomicNo(i) !== 6) continue;
        if (molecule.getStereoBond(i) >= 0) continue;
        const neighbourSymmetries = getNeighbourSymmetries(molecule, i);
        if (neighbourSymmetries.length <= 2) continue;
        const stereoBond = molecule.getAtomPreferredStereoBond(i);
        if (stereoBond !== -1) {
          molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
          if (molecule.getBondAtom(1, stereoBond) === i) {
            const connAtom = molecule.getBondAtom(0, stereoBond);
            molecule.setBondAtom(0, stereoBond, i);
            molecule.setBondAtom(1, stereoBond, connAtom);
          }
          // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0);
        }
      }
    }
    function getNeighbourSymmetries(molecule, iAtom) {
      const neighbourSymmetries = [];
      for (let j = 0; j < molecule.getAllConnAtoms(iAtom); j++) {
        const connAtom = molecule.getConnAtom(iAtom, j);
        const symmetryRank = molecule.getSymmetryRank(connAtom);
        if (!neighbourSymmetries.includes(symmetryRank)) {
          neighbourSymmetries.push(molecule.getSymmetryRank(connAtom));
        }
      }
      return neighbourSymmetries;
    }

    /**
     * This function will add missing chiral bonds on carbons ensure that all enantiotopic
     * or diastereotopic atoms can be identified uniquely
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {number} [options.esrType=Molecule.cESRTypeAnd]
     * @param {boolean} [options.atLeastThreeAtoms=true] - if true, only carbons with at least three atoms will be considered
     */
    function ensureHeterotopicChiralBonds(molecule, options = {}) {
      const {
        Molecule
      } = molecule.getOCL();
      const {
        esrType = Molecule.cESRTypeAnd,
        atLeastThreeAtoms = true
      } = options;
      molecule.ensureHelperArrays(Molecule.cHelperBitNeighbours);
      const heterotopicCarbons = getChiralOrHeterotopicCarbons(molecule);
      for (const i of heterotopicCarbons) {
        if (atLeastThreeAtoms && molecule.getAllConnAtoms(i) < 3) continue;
        if (molecule.getStereoBond(i) === -1) {
          const stereoBond = molecule.getAtomPreferredStereoBond(i);
          if (stereoBond !== -1) {
            molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
            if (molecule.getBondAtom(1, stereoBond) === i) {
              const connAtom = molecule.getBondAtom(0, stereoBond);
              molecule.setBondAtom(0, stereoBond, i);
              molecule.setBondAtom(1, stereoBond, connAtom);
            }
            // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
            molecule.setAtomESR(i, esrType, 0);
          }
        }
      }
    }

    /**
     * Expand all the implicit hydrogens and ensure chiral bonds on heterotopic bonds
     * @param molecule
     * @param options
     * @returns
     */
    function getMoleculeWithH(molecule, options) {
      const {
        logger,
        maxNbAtoms
      } = options;
      const moleculeWithH = molecule.getCompactCopy();
      moleculeWithH.addImplicitHydrogens();
      if (moleculeWithH.getAllAtoms() > maxNbAtoms) {
        logger.warn(`too many atoms to evaluate heterotopic chiral bonds: ${moleculeWithH.getAllAtoms()} > ${maxNbAtoms}`);
      } else {
        ensureHeterotopicChiralBonds(moleculeWithH);
      }
      return moleculeWithH;
    }

    /**
     * In order to be able to give a unique ID to all the atoms we are replacing the H by X
     * @param moleculeWithH
     * @returns
     */
    function getXMolecule(moleculeWithH) {
      const xAtomNumber = getXAtomicNumber(moleculeWithH);
      const xMolecule = moleculeWithH.getCompactCopy();
      for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (xMolecule.getAtomicNo(i) === 1) {
          xMolecule.setAtomicNo(i, xAtomNumber);
        }
      }
      return xMolecule;
    }

    /**
     * This class deals with topicity information and hose codes
     * It is optimized to avoid recalculation of the same information
     */
    class TopicMolecule {
      originalMolecule;
      molecule;
      idCode;
      options;
      cache;
      constructor(molecule, options = {}) {
        this.originalMolecule = molecule;
        this.options = {
          maxPathLength: 5,
          maxNbAtoms: 250,
          logger: console,
          ...options
        };
        this.idCode = molecule.getIDCode();
        this.molecule = this.originalMolecule.getCompactCopy();
        this.molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
        this.molecule.ensureHelperArrays(this.molecule.getOCL().Molecule.cHelperNeighbours);
        this.cache = {};
      }
      /**
       * This method ensures that all the atoms have a mapNo corresponding to the atom number.
       * It will enforce mapNo in molecule and moleculeWithH
       * We start numbering the atoms at 1
       */
      setAtomNoInMapNo() {
        const molecules = [this.molecule, this.moleculeWithH];
        for (const molecule of molecules) {
          for (let i = 0; i < molecule.getAllAtoms(); i++) {
            molecule.setAtomMapNo(i, i + 1, false);
          }
        }
      }
      /**
       * This method ensures that all the atoms have a mapNo in the molecule (and not the moleculeWithH! )
       */
      ensureMapNo() {
        const existingMapNo = {};
        for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
          const mapNo = this.molecule.getAtomMapNo(i);
          if (mapNo) {
            if (existingMapNo[mapNo]) {
              throw new Error('The molecule contains several atoms with the same mapNo');
            }
            existingMapNo[mapNo] = true;
          }
        }
        let nextMapNo = 1;
        for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
          const mapNo = this.molecule.getAtomMapNo(i);
          if (!mapNo) {
            while (existingMapNo[nextMapNo]) {
              nextMapNo++;
            }
            existingMapNo[nextMapNo] = true;
            this.molecule.setAtomMapNo(i, nextMapNo, false);
          }
        }
      }
      /**
       * For each atom we will return an array of objects that contains the
       * different possible path as well as the canonic hose codes
       * @param options
       * @returns
       */
      getHoseCodesForPath(options = {}) {
        return getCanonizedHoseCodesForPath(this, options);
      }
      /**
       * Return one fragment for a specific sphere size and specific root atoms
       * @param rootAtoms
       * @param options
       * @returns
       */
      getHoseFragment(rootAtoms, options = {}) {
        const {
          sphereSize = 2,
          tagAtoms = rootAtoms,
          tagAtomFct
        } = options;
        const fragments = getHoseCodesForAtomsAsFragments(this.moleculeWithH, {
          rootAtoms,
          minSphereSize: sphereSize,
          maxSphereSize: sphereSize,
          tagAtoms,
          tagAtomFct
        });
        return fragments[0];
      }
      getAtomPathsFrom(atom, options = {}) {
        const {
          minPathLength = 1,
          maxPathLength = this.options.maxPathLength,
          toAtomicNo
        } = options;
        if (maxPathLength > this.options.maxPathLength) {
          throw new Error('The maxPathLength is too long, you should increase the maxPathLength when instantiating the TopicMolecule');
        }
        const atomPaths = this.atomsPaths[atom];
        if (!atomPaths) {
          throw new Error('Unexpected missing atom path');
        }
        const paths = [];
        for (let i = minPathLength; i <= maxPathLength; i++) {
          const atomPathValue = atomPaths[i];
          if (!atomPathValue) {
            throw new Error(`Unexpected missing atom path at index ${i}`);
          }
          for (const atomPath of atomPathValue) {
            if (!toAtomicNo || this.moleculeWithH.getAtomicNo(atomPath.path.at(-1)) === toAtomicNo) {
              paths.push(atomPath.path);
            }
          }
        }
        return paths;
      }
      getAtomPaths(atom1, atom2, options = {}) {
        const {
          pathLength
        } = options;
        if (pathLength !== undefined && pathLength > this.options.maxPathLength) {
          throw new Error('The distance is too long, you should increase the maxPathLength when instantiating the TopicMolecule');
        }
        const atomPaths = this.atomsPaths[atom1];
        if (!atomPaths) {
          throw new Error('Unexpected missing atom path');
        }
        const minDistance = pathLength || 0;
        const maxDistance = pathLength || this.options.maxPathLength;
        const paths = [];
        for (let i = minDistance; i <= maxDistance; i++) {
          const atomPathValue = atomPaths[i];
          if (!atomPathValue) {
            throw new Error(`Unexpected missing atom path at index ${i}`);
          }
          for (const atomPath of atomPathValue) {
            if (atomPath.path.at(-1) === atom2) {
              paths.push(atomPath.path);
            }
          }
        }
        return paths;
      }
      get atomsPaths() {
        if (this.cache.atomsPaths) return this.cache.atomsPaths;
        this.cache.atomsPaths = getAllAtomsPaths(this.moleculeWithH, {
          maxPathLength: this.options.maxPathLength
        });
        return this.cache.atomsPaths;
      }
      toMolfile(options = {}) {
        const {
          version = 2
        } = options;
        if (version === 2) {
          return this.molecule.toMolfile();
        }
        return this.molecule.toMolfileV3();
      }
      getMolecule() {
        return this.molecule;
      }
      /**
       * Returns a new TopicMolecule but will copy precalculated information
       * if possible (same idCode). This is very practical when expanding hydrogens
       * for example.
       * @param molecule
       * @returns
       */
      fromMolecule(molecule) {
        const idCode = molecule.getIDCode();
        if (idCode !== this.idCode) {
          // no way for optimisation
          return new TopicMolecule(molecule);
        }
        const topicMolecule = new TopicMolecule(molecule);
        topicMolecule.cache = {
          canonizedDiaIDs: this.cache.canonizedDiaIDs,
          canonizedHoseCodes: this.cache.canonizedHoseCodes
        };
        return topicMolecule;
      }
      /**
       * Returns a molecule with all the hydrogens added. The order is NOT canonized
       */
      get moleculeWithH() {
        if (this.cache.moleculeWithH) return this.cache.moleculeWithH;
        this.cache.moleculeWithH = getMoleculeWithH(this.molecule, {
          maxNbAtoms: this.options.maxNbAtoms,
          logger: this.options.logger
        });
        return this.cache.moleculeWithH;
      }
      get xMolecule() {
        if (this.cache.xMolecule) return this.cache.xMolecule;
        this.cache.xMolecule = getXMolecule(this.moleculeWithH);
        return this.cache.xMolecule;
      }
      /**
       * This is related to the current moleculeWithH. The order is NOT canonized
       */
      get diaIDs() {
        if (this.cache.diaIDs) return this.cache.diaIDs;
        const diaIDs = [];
        if (this.moleculeWithH.getAllAtoms() > this.options.maxNbAtoms) {
          this.options.logger.warn(`too many atoms to evaluate heterotopicity: ${this.moleculeWithH.getAllAtoms()} > ${this.options.maxNbAtoms}`);
        } else {
          for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
            diaIDs.push(this.canonizedDiaIDs[this.finalRanks[i]]);
          }
        }
        this.cache.diaIDs = diaIDs;
        return diaIDs;
      }
      /**
       * We return the atomIDs corresponding to the specified diaID as well has the attached hydrogens or heavy atoms
       * @param diaID
       * @returns
       */
      getDiaIDsObject() {
        return groupDiastereotopicAtomIDsAsObject(this.diaIDs, this.molecule, this.moleculeWithH);
      }
      /**
       * This is related to the current moleculeWithH. The order is NOT canonized
       */
      get hoseCodes() {
        if (this.cache.hoseCodes) return this.cache.hoseCodes;
        const hoseCodes = [];
        for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
          hoseCodes.push(this.canonizedHoseCodes[this.finalRanks[i]]);
        }
        this.cache.hoseCodes = hoseCodes;
        return hoseCodes;
      }
      get canonizedDiaIDs() {
        if (this.cache.canonizedDiaIDs) return this.cache.canonizedDiaIDs;
        this.cache.canonizedDiaIDs = getCanonizedDiaIDs(this, {
          maxNbAtoms: this.options.maxNbAtoms,
          logger: this.options.logger
        });
        return this.cache.canonizedDiaIDs;
      }
      get canonizedHoseCodes() {
        if (this.cache.canonizedHoseCodes) {
          return this.cache.canonizedHoseCodes;
        }
        this.cache.canonizedHoseCodes = getCanonizedHoseCodes(this);
        return this.cache.canonizedHoseCodes;
      }
      /**
       * Returns the distance matrix for the current moleculeWithH
       */
      get distanceMatrix() {
        return getConnectivityMatrix(this.moleculeWithH, {
          pathLength: true
        });
      }
      get diaIDsAndInfo() {
        if (this.cache.diaIDsAndInfo) {
          return this.cache.diaIDsAndInfo;
        }
        this.cache.diaIDsAndInfo = getDiaIDsAndInfo(this, this.canonizedDiaIDs);
        return this.cache.diaIDsAndInfo;
      }
      /**
       * Returns symmetryRanks for all the atoms including hydrogens. Those ranks
       * deals with topicity and is related to the current moleculeWithH.
       * In order to calculate the ranks we replace all the
       * hydrogens with a X atom.
       */
      get heterotopicSymmetryRanks() {
        if (this.cache.heterotopicSymmetryRanks) {
          return this.cache.heterotopicSymmetryRanks;
        }
        this.cache.heterotopicSymmetryRanks = getHeterotopicSymmetryRanks(this.xMolecule);
        return [...this.cache.heterotopicSymmetryRanks];
      }
      /**
       * Returns finalRanks for all the atoms including hydrogens. Those ranks
       * deals with topicity and is related to the current moleculeWithH.
       * All the atoms have a unique identifier.j
       * In order to calculate the ranks we replace all the
       * hydrogens with a X atom.
       */
      get finalRanks() {
        if (this.cache.finalRanks) return this.cache.finalRanks;
        this.cache.finalRanks = getFinalRanks(this.xMolecule);
        return this.cache.finalRanks;
      }
      toMolfileWithH(options = {}) {
        const {
          version = 2
        } = options;
        if (version === 2) {
          return this.moleculeWithH.toMolfile();
        }
        return this.moleculeWithH.toMolfileV3();
      }
      toMolfileWithoutH(options = {}) {
        const molecule = this.molecule.getCompactCopy();
        molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
        const atomsToDelete = [];
        for (let atomID = 0; atomID < molecule.getAtoms(); atomID++) {
          if (molecule.getExplicitHydrogens(atomID) > 0) {
            for (let i = 0; i < molecule.getAllConnAtoms(atomID); i++) {
              const connectedAtom = molecule.getConnAtom(atomID, i);
              if (molecule.getAtomicNo(connectedAtom) === 1) {
                atomsToDelete.push(connectedAtom);
              }
            }
          }
        }
        molecule.deleteAtoms(atomsToDelete);
        const {
          version = 2
        } = options;
        if (version === 2) {
          return molecule.toMolfile();
        }
        return molecule.toMolfileV3();
      }
      /**
       * Returns an array of objects containing the oclID and the corresponding hydrogens and atoms
       * for the specified atomLabel (if any)
       * This always applied to the molecule with expanded hydrogens and chirality
       * @param options
       * @returns
       */
      getGroupedDiastereotopicAtomIDs(options = {}) {
        if (!this.diaIDs) return undefined;
        return groupDiastereotopicAtomIDs$1(this.diaIDs, this.moleculeWithH, options);
      }
      /**
       * This method returns a mapping between the diaIDs of the current molecule.
       * It expects that the initial molfile and the final molfile contains atomMapNo
       * in order to track which atom becomes which one.
       * @param molecule
       */
      getDiaIDsMapping(molecule) {
        const topicMolecule = new TopicMolecule(molecule);
        const originalDiaIDs = this.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
        const destinationDiaIDs = topicMolecule.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
        const mapping = {};
        // we first check all the atoms present in the molfile
        for (const destinationDiaID of destinationDiaIDs) {
          const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
          const newIDCode = destinationDiaID.idCode;
          const oldIDCode = originalDiaID.idCode;
          if (oldIDCode in mapping) {
            if (mapping[oldIDCode] !== newIDCode) {
              mapping[oldIDCode] = undefined;
            }
          } else {
            mapping[oldIDCode] = newIDCode;
          }
        }
        // we now check all the attached hydrogens that are not defined in the molfile and were not yet mapped
        for (const destinationDiaID of destinationDiaIDs) {
          const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
          for (let i = 0; i < originalDiaID.attachedHydrogensIDCodes.length; i++) {
            const oldHydrogenIDCode = originalDiaID.attachedHydrogensIDCodes.at(i);
            if (mapping[oldHydrogenIDCode]) continue;
            const newHydrogenIDCode = destinationDiaID.attachedHydrogensIDCodes[i];
            if (oldHydrogenIDCode && newHydrogenIDCode) {
              if (oldHydrogenIDCode in mapping) {
                if (mapping[oldHydrogenIDCode] !== newHydrogenIDCode) {
                  mapping[oldHydrogenIDCode] = undefined;
                }
              } else {
                mapping[oldHydrogenIDCode] = newHydrogenIDCode;
              }
            }
          }
        }
        return mapping;
      }
    }
    function groupDiastereotopicAtomIDs$1(diaIDs, molecule, options = {}) {
      const diaIDsObject = groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, molecule, options);
      return Object.values(diaIDsObject);
    }
    function groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, moleculeWithH, options = {}) {
      const {
        atomLabel
      } = options;
      const diaIDsObject = {};
      for (let i = 0; i < diaIDs.length; i++) {
        if (!atomLabel || moleculeWithH.getAtomLabel(i) === atomLabel) {
          const diaID = diaIDs[i];
          if (!diaIDsObject[diaID]) {
            diaIDsObject[diaID] = {
              counter: 0,
              atoms: [],
              oclID: diaID,
              atomLabel: moleculeWithH.getAtomLabel(i),
              heavyAtoms: [],
              attachedHydrogens: [],
              existingAtoms: []
            };
          }
          if (moleculeWithH.getAtomicNo(i) === 1) {
            const connected = moleculeWithH.getConnAtom(i, 0);
            if (!diaIDsObject[diaID].heavyAtoms.includes(connected)) {
              diaIDsObject[diaID].heavyAtoms.push(connected);
            }
            if (molecule.getAtomicNo(i)) {
              diaIDsObject[diaID].existingAtoms.push(i);
            } else if (!diaIDsObject[diaID].existingAtoms.includes(connected)) {
              diaIDsObject[diaID].existingAtoms.push(connected);
            }
          } else {
            for (let j = 0; j < moleculeWithH.getAllConnAtoms(i); j++) {
              const connected = moleculeWithH.getConnAtom(i, j);
              if (moleculeWithH.getAtomicNo(connected) === 1) {
                diaIDsObject[diaID].attachedHydrogens.push(connected);
              }
            }
            diaIDsObject[diaID].existingAtoms.push(i);
          }
          diaIDsObject[diaID].counter++;
          diaIDsObject[diaID].atoms.push(i);
        }
      }
      for (const diaIDObject of Object.values(diaIDsObject)) {
        diaIDObject.existingAtoms.sort((a, b) => a - b);
        diaIDObject.attachedHydrogens.sort((a, b) => a - b);
      }
      return diaIDsObject;
    }

    /**
     * Returns an array of symmetry ranks.
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @returns
     */
    function getSymmetryRanks(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      const xAtomNumber = getXAtomicNumber(molecule);
      // most of the molecules have some symmetry
      const internalMolecule = molecule.getCompactCopy();
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (internalMolecule.getAtomicNo(i) === 1) {
          internalMolecule.setAtomicNo(i, xAtomNumber);
        }
      }
      internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const symmetryRanks = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        symmetryRanks.push(internalMolecule.getSymmetryRank(i));
      }
      return symmetryRanks;
    }

    /**
     * Returns an array of diastereotopic ID (as idCode)
     * @param {import('openchemlib').Molecule} molecule
     */
    function getDiastereotopicAtomIDs(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      ensureHeterotopicChiralBonds(molecule);
      const symmetryRanks = getSymmetryRanks(molecule);
      const numberAtoms = molecule.getAllAtoms();
      const ids = [];
      const cache = {};
      for (let iAtom = 0; iAtom < numberAtoms; iAtom++) {
        const rank = symmetryRanks[iAtom];
        if (rank && cache[rank]) {
          ids[iAtom] = cache[rank];
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(molecule);
        tagAtom(tempMolecule, iAtom);
        makeRacemic(tempMolecule);
        // We need to ensure the helper array in order to get correctly the result of racemisation
        ids[iAtom] = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        cache[rank] = ids[iAtom];
      }
      return ids;
    }

    /**
     *
     * @param {import('openchemlib').Molecule} molecule
     */
    function getDiastereotopicAtomIDsAndH(molecule) {
      const OCL = molecule.getOCL();
      molecule = molecule.getCompactCopy();
      molecule.addImplicitHydrogens();
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      const diaIDs = getDiastereotopicAtomIDs(molecule);
      const newDiaIDs = [];
      for (let i = 0; i < diaIDs.length; i++) {
        const diaID = diaIDs[i];
        const newDiaID = {
          oclID: diaID,
          hydrogenOCLIDs: [],
          nbHydrogens: 0
        };
        if (molecule.getAtomicNo(i) === 1) {
          const atom = molecule.getConnAtom(i, 0);
          newDiaID.heavyAtom = diaIDs[atom];
        }
        for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
          const atom = molecule.getConnAtom(i, j);
          if (molecule.getAtomicNo(atom) === 1) {
            newDiaID.nbHydrogens++;
            if (!newDiaID.hydrogenOCLIDs.includes(diaIDs[atom])) {
              newDiaID.hydrogenOCLIDs.push(diaIDs[atom]);
            }
          }
        }
        newDiaIDs.push(newDiaID);
      }
      return newDiaIDs;
    }

    function groupDiastereotopicAtomIDs(diaIDs, molecule, options = {}) {
      const {
        atomLabel
      } = options;
      const diaIDsObject = {};
      for (let i = 0; i < diaIDs.length; i++) {
        if (!atomLabel || molecule.getAtomLabel(i) === atomLabel) {
          const diaID = diaIDs[i];
          if (!diaIDsObject[diaID]) {
            diaIDsObject[diaID] = {
              counter: 0,
              atoms: [],
              oclID: diaID,
              atomLabel: molecule.getAtomLabel(i)
            };
          }
          diaIDsObject[diaID].counter++;
          diaIDsObject[diaID].atoms.push(i);
        }
      }
      return Object.keys(diaIDsObject).map(key => diaIDsObject[key]);
    }

    /**
     * This function groups the diasterotopic atomIds of the molecule based on equivalence of atoms. The output object contains
     * a set of chemically equivalent atoms(element.atoms) and the groups of magnetically equivalent atoms (element.magneticGroups)
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {string} [options.atomLabel] - Select atoms of the given atomLabel. By default it returns all the explicit atoms in the molecule
     * @returns {Array}
     */
    function getGroupedDiastereotopicAtomIDs(molecule, options = {}) {
      const diaIDs = getDiastereotopicAtomIDs(molecule);
      return groupDiastereotopicAtomIDs(diaIDs, molecule, options);
    }

    /**
     * Parse a molfile and returns an object containing the molecule, the map and the diaIDs
     * The map allows to reload properties assigned to the atom molfile
     * Please take care than numbering of atoms starts at 0 !
     * @param {typeof import('openchemlib')} OCL - openchemlib library
     * @param {string} molfile
     * @returns
     */
    function getDiastereotopicAtomIDsFromMolfile(OCL, molfile) {
      const {
        map,
        molecule
      } = OCL.Molecule.fromMolfileWithAtomMap(molfile);
      const diaIDsArray = getDiastereotopicAtomIDsAndH(molecule);
      const diaIDs = {};
      for (let i = 0; i < map.length; i++) {
        diaIDs[map[i]] = {
          source: map[i],
          destination: i,
          ...diaIDsArray[i]
        };
      }
      return {
        map: diaIDs,
        molecule,
        diaIDs: diaIDsArray
      };
    }

    /**
     * Returns a SVG
     * @param {*} molecule
     * @param {*} [options={}]
     */
    function toDiastereotopicSVG(molecule, options = {}) {
      const {
        width = 300,
        height = 200,
        prefix = 'ocl',
        heavyAtomHydrogen = false
      } = options;
      let svg = options.svg;
      let diaIDs = [];
      const hydrogenInfo = {};
      for (const line of getDiastereotopicAtomIDsAndH(molecule)) {
        hydrogenInfo[line.oclID] = line;
      }
      if (heavyAtomHydrogen) {
        for (let i = 0; i < molecule.getAtoms(); i++) {
          diaIDs.push([]);
        }
        const groupedDiaIDs = molecule.getGroupedDiastereotopicAtomIDs();
        for (const diaID of groupedDiaIDs) {
          if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
            for (const atom of diaID.atoms) {
              for (const id of hydrogenInfo[diaID.oclID].hydrogenOCLIDs) {
                if (!diaIDs[atom * 1].includes(id)) diaIDs[atom].push(id);
              }
            }
          }
        }
      } else {
        diaIDs = molecule.getDiastereotopicAtomIDs().map(a => [a]);
      }
      if (!svg) svg = molecule.toSVG(width, height, prefix);
      svg = svg.replaceAll(/Atom:\d+"/g, value => {
        const atom = value.replaceAll(/\D/g, '');
        return `${value} data-diaid="${diaIDs[atom].join(',')}"`;
      });
      return svg;
    }

    /**
     * Returns the hose codes for all atoms in the molecule
     * @param {*} molecule
     * @param {object} [options={}]
     * @param {string[]} [options.atomLabels]
     * @param {number} [options.minSphereSize=0]
     * @param {number} [options.maxSphereSize=4]
     * @returns
     */
    function getHoseCodes(molecule, options = {}) {
      const {
        atomLabels,
        minSphereSize,
        maxSphereSize
      } = options;
      const {
        Molecule
      } = molecule.getOCL();
      const atomicNumbers = atomLabels?.map(label => Molecule.getAtomicNoFromLabel(label));
      const internalMolecule = getCompactCopyWithoutCustomLabels(molecule);
      internalMolecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(internalMolecule);
      const hoses = [];
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        if (atomicNumbers && !atomicNumbers.includes(internalMolecule.getAtomicNo(i))) {
          hoses.push(undefined);
        } else {
          const tempMolecule = internalMolecule.getCompactCopy();
          tagAtom(tempMolecule, i);
          hoses.push(getHoseCodesForAtomsAsStrings(tempMolecule, {
            minSphereSize,
            maxSphereSize
          }));
        }
      }
      return hoses;
    }

    /**
     * Returns an object containing a molfile, molfile with hydrogens, hoses codes and optionally the diaIDs
     * and the diaIDs
     * The map allows to reload properties assigned to the atom molfile
     * Please take care than numbering of atoms starts at 0 !
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {string[]} [options.atomLabels]
     * @param {boolean} [options.calculateDiastereotopicIDs=true]
     * @param {number} [options.minSphereSize=0]
     * @param {number} [options.maxSphereSize=4]
     * @returns
     */
    function getHoseCodesAndInfo(molecule, options = {}) {
      const {
        minSphereSize,
        maxSphereSize,
        calculateDiastereotopicIDs = true
      } = options;
      const {
        Molecule
      } = molecule.getOCL();
      molecule = molecule.getCompactCopy();
      // this will force reordering of the hydrogens to the end, just to have the same order as in the molfile
      molecule.ensureHelperArrays(Molecule.cHelperNeighbours);
      const newMolfile = molecule.toMolfile();
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const newMolfileWithH = molecule.toMolfile();
      const symmetryRanks = getSymmetryRanks(molecule);
      const cache = {};
      const hoses = [];
      const diaIDs = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        const rank = symmetryRanks[i];
        if (rank && cache[rank]) {
          diaIDs.push(cache[rank].diaID);
          hoses.push(cache[rank].hose);
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(molecule);
        tagAtom(tempMolecule, i);
        let diaID;
        if (calculateDiastereotopicIDs) {
          makeRacemic(tempMolecule);
          diaID = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
          diaIDs.push(diaID);
        }
        const hose = getHoseCodesForAtomsAsStrings(tempMolecule, {
          minSphereSize,
          maxSphereSize
        });
        hoses.push(hose);
        cache[rank] = {
          diaID,
          hose
        };
      }
      const distanceMatrix = getConnectivityMatrix(molecule, {
        pathLength: true
      });
      return {
        molfile: newMolfile,
        molfileWithH: newMolfileWithH,
        hoses,
        diaIDs: calculateDiastereotopicIDs ? diaIDs : undefined,
        moleculeWithHydrogens: molecule,
        distanceMatrix
      };
    }

    const FULL_HOSE_CODE = 1;
    const HOSE_CODE_CUT_C_SP3_SP3 = 2;
    /**
     * Returns the hose code for specific atom numbers
     * @param {import('openchemlib').Molecule} originalMolecule - The OCL molecule to be fragmented
     * @param {number[]} rootAtoms
     * @param {object} [options={}]
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtoms(originalMolecule, rootAtoms = [], options = {}) {
      const {
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE
      } = options;
      const molecule = getCompactCopyWithoutCustomLabels(originalMolecule);
      // those 2 lines should be done only once
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      const allowedCustomLabels = [];
      for (const rootAtom of rootAtoms) {
        allowedCustomLabels.push(tagAtom(molecule, rootAtom));
      }
      return getHoseCodesForAtomsAsStrings(molecule, {
        minSphereSize,
        maxSphereSize,
        allowedCustomLabels,
        kind
      });
    }

    /**
     * Returns the hose code for a specific atom number
     * @param {import('openchemlib').Molecule} originalMolecule
     * @param {number} rootAtom
     * @param {object} [options={}]
     * @param {boolean} [options.isTagged] - Specify is the atom is already tagged
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtom(originalMolecule, rootAtom, options = {}) {
      return getHoseCodesForAtoms(originalMolecule, [rootAtom], options);
    }

    /**
     * Returns an array of strings (idCodes) specified molecule. Each string corresponds to a
     * hose code. By default it will calculate the hose codes for sphere 0 to 4 and will reuse
     * the existing tagged atoms.
     * This method ensure implicity hydrogens and possible missing chiral bonds.
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to process.
     * @param {object} options - Options for generating hose codes.
     * @param {string[]} [options.allowedCustomLabels] - Array of the custom labels that are considered as root atoms. By default all atoms having a customLabel
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     *  @param {number[]} [options.rootAtoms=[]] - Array of atom from which we should start to create the HOSE. By default we will used the taggedAtoms
     * @param {number[]} [options.tagAtoms=[]] - Array of atom indices to tag as root atoms
     * @param {Function} [options.tagAtomFct=tagAtom] - Function to tag an atom as root atom. By default it is defined internal
     * @returns {Array} - An array of hose code fragments.
     */
    function getHoseCodesFromDiastereotopicID(molecule, options = {}) {
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      return getHoseCodesForAtomsAsStrings(molecule, options);
    }

    let fragment$1;
    /**
     * Returns the hose code for a specific atom number
     * @param {import('openchemlib').Molecule} molecule
     * @param from
     * @param to
     * @param maxLength
     */
    function getHoseCodesForPath(molecule, from, to, maxLength) {
      const OCL = molecule.getOCL();
      const originalFrom = from;
      const originalTo = to;
      molecule = getCompactCopyWithoutCustomLabels(molecule);
      const originalAtoms = []; // path before renumbering
      molecule.getPath(originalAtoms, from, to, maxLength + 1);
      let torsion;
      if (originalAtoms.length === 4) {
        torsion = molecule.calculateTorsion(originalAtoms);
      }
      const tag1 = tagAtom(molecule, from);
      const tag2 = tagAtom(molecule, to);
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      from = -1;
      to = -1;
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (tag1 === tag2) {
          if (molecule.getAtomCustomLabel(i) === tag1) {
            if (from === -1) {
              from = i;
            } else {
              to = i;
            }
          }
        } else {
          if (tag1 === molecule.getAtomCustomLabel(i)) {
            from = i;
          }
          if (tag2 === molecule.getAtomCustomLabel(i)) {
            to = i;
          }
        }
      }
      if (!fragment$1) fragment$1 = new OCL.Molecule(0, 0);
      const atoms = [];
      molecule.getPath(atoms, from, to, maxLength + 1);
      let min = 0;
      let max = 0;
      const atomMask = new Array(molecule.getAllAtoms()).fill(false);
      const atomList = new Array(molecule.getAllAtoms()).fill(-1);
      const hoses = [];
      for (let sphere = 0; sphere <= 2; sphere++) {
        if (max === 0) {
          for (const atom of atoms) {
            atomMask[atom] = true;
            atomList[max++] = atom;
          }
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            const atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              const connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                atomMask[connAtom] = true;
                atomList[newMax++] = connAtom;
              }
            }
          }
          min = max;
          max = newMax;
        }
        const atomMap = [];
        molecule.copyMoleculeByAtoms(fragment$1, atomMask, true, atomMap);
        makeRacemic(fragment$1);
        const oclID = fragment$1.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        hoses.push({
          sphere,
          oclID
        });
      }
      return {
        atoms: originalAtoms,
        from: originalFrom,
        to: originalTo,
        torsion,
        hoses,
        length: originalAtoms.length - 1
      };
    }

    /**
     * Create a polymer from a unit, alpha and omega groups
     * A unit must contain a R1 and a R2
     * An alpha end group must contain a R1
     * An omega end group must contain a R2
     * @param {import('openchemlib').Molecule} unit - an instance of OCL.Molecule
     * @param {object} options
     * @param {number} [options.count=10] - number of units
     * @param {boolean} [options.markMonomer=false] - mark the different units of the polymer in the atom map
     * @param {import('openchemlib').Molecule} [options.alpha] - alpha end group, default is an hydrogen
     * @param {import('openchemlib').Molecule} [options.gamma] - omega end group, default is an hydrogen
     */
    function createPolymer(unit, options = {}) {
      const {
        count = 10
      } = options;
      checkEntity(unit, 'unit');
      const {
        Molecule
      } = unit.getOCL();
      const {
        alpha,
        gamma
      } = getAlphaGamma(unit, options);
      checkEntity(alpha, 'alpha');
      checkEntity(gamma, 'gamma');
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      const polymer = alpha.getCompactCopy();
      polymer.addMolecule(getUnit(unit, 1, options));
      addBond(polymer, r1AtomicNo, r1AtomicNo);
      for (let i = 0; i < count - 1; i++) {
        polymer.addMolecule(getUnit(unit, i + 2, options));
        addBond(polymer, r2AtomicNo, r1AtomicNo);
      }
      polymer.addMolecule(gamma);
      addBond(polymer, r2AtomicNo, r2AtomicNo);
      polymer.ensureHelperArrays(Molecule.cHelperNeighbours);
      // encoding directly in atomNapNo didn't work out because it was removed when deleting atoms
      for (let i = 0; i < polymer.getAtoms(); i++) {
        polymer.setAtomMapNo(i, (polymer.getAtomCustomLabel(i) || '').replace(/monomer_/, ''));
        polymer.setAtomCustomLabel(i, '');
      }
      return polymer;
    }
    function getUnit(unit, index, options) {
      const {
        markMonomer = false
      } = options;
      if (markMonomer) {
        unit = unit.getCompactCopy();
        unit.ensureHelperArrays(unit.getOCL().Molecule.cHelperNeighbours);
        for (let j = 0; j < unit.getAtoms(); j++) {
          unit.setAtomCustomLabel(j, `monomer_${index}`);
        }
      }
      return unit;
    }
    function addBond(molecule, firstAtomicNo, secondAtomicNo) {
      molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
      let i, j;
      loop: for (i = 0; i < molecule.getAtoms(); i++) {
        if (molecule.getAtomicNo(i) === firstAtomicNo) {
          for (j = i + 1; j < molecule.getAtoms(); j++) {
            if (molecule.getAtomicNo(j) === secondAtomicNo) {
              molecule.addBond(molecule.getConnAtom(i, 0), molecule.getConnAtom(j, 0), 1);
              break loop;
            }
          }
        }
      }
      molecule.deleteAtoms([i, j]);
    }
    function checkEntity(unit, kind) {
      let nbR1 = 1;
      let nbR2 = 1;
      switch (kind) {
        case 'unit':
          break;
        case 'alpha':
          nbR2 = 0;
          break;
        case 'gamma':
          nbR1 = 0;
          break;
        default:
          throw new Error('Unknown kind');
      }
      if (!unit) {
        throw new Error('unit is required');
      }
      const {
        Molecule
      } = unit.getOCL();
      // unit must contain ONE R1 and ONE R2
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      let r1Count = 0;
      let r2Count = 0;
      for (let i = 0; i < unit.getAtoms(); i++) {
        if (unit.getAtomicNo(i) === r1AtomicNo) {
          r1Count++;
        }
        if (unit.getAtomicNo(i) === r2AtomicNo) {
          r2Count++;
        }
      }
      if (r1Count !== nbR1) {
        throw new Error(`${kind} must contain ${nbR1} R1`);
      }
      if (r2Count !== nbR2) {
        throw new Error(`${kind} must contain ${nbR2} R2`);
      }
    }
    function getAlphaGamma(unit, options) {
      let {
        alpha,
        gamma
      } = options;
      const {
        Molecule
      } = unit.getOCL();
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      if (!alpha) {
        alpha = Molecule.fromSmiles('CC');
        alpha.setAtomicNo(0, r1AtomicNo);
        alpha.setAtomicNo(1, 1);
      }
      if (!gamma) {
        gamma = Molecule.fromSmiles('CC');
        gamma.setAtomicNo(0, r2AtomicNo);
        gamma.setAtomicNo(1, 1);
      }
      return {
        alpha,
        gamma
      };
    }
    function getR1R2AtomicNo(Molecule) {
      const r1AtomicNo = Molecule.getAtomicNoFromLabel('R1', Molecule.cPseudoAtomsRGroups);
      const r2AtomicNo = Molecule.getAtomicNoFromLabel('R2', Molecule.cPseudoAtomsRGroups);
      return {
        r1AtomicNo,
        r2AtomicNo
      };
    }

    var IP=class{copyMolecule(ft){let Bt=ft.getIDCodeAndCoordinates();navigator.clipboard.writeText(`${Bt.idCode} ${Bt.coordinates}`);}pasteMolecule(){return null;}};var hu="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",SA=new Uint8Array(256);for(let j=0;j<hu.length;j++)SA[hu.charCodeAt(j)]=j;function Tu(j){let ft=j.length*.75,Bt=j.length,dt,It=0,v1,Vt,e1,t1;j.at(-1)==="="&&(ft--,j.at(-2)==="="&&ft--);let g1=new ArrayBuffer(ft),c1=new Uint8Array(g1);for(dt=0;dt<Bt;dt+=4)v1=SA[j.charCodeAt(dt)],Vt=SA[j.charCodeAt(dt+1)],e1=SA[j.charCodeAt(dt+2)],t1=SA[j.charCodeAt(dt+3)],c1[It++]=v1<<2|Vt>>4,c1[It++]=(Vt&15)<<4|e1>>2,c1[It++]=(e1&3)<<6|t1&63;return g1;}function xP(j){return j.toString(16).padStart(2,"0");}var _e=globalThis.devicePixelRatio||1;var $n=class{constructor(ft){this.ctx=ft,this.ctx.textAlign="left",this.ctx.textBaseline="top",this.currentFontSize=12,this.currentFont="12px sans-serif",this.ctx.font=this.currentFont,this.currentColor="#000000",this.currentLineWidth=1,this.canvasCache=new Map();}clearRect(ft,Bt,dt,It){this.ctx.clearRect(ft,Bt,dt,It);}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}getSelectionBackgroundRGB(){return 12310268;}getLineWidth(){return this.currentLineWidth;}setRGB(ft){let Bt=ft>>>16&255,dt=ft>>>8&255,It=ft>>>0&255;this.currentColor=`#${xP(Bt)}${xP(dt)}${xP(It)}`,this.ctx.fillStyle=this.currentColor,this.ctx.strokeStyle=this.currentColor;}setFont(ft,Bt,dt){this.currentFontSize=ft,this.currentFont=`${Bt?"bold":""} ${dt?"italic":""} ${ft}px sans-serif`,this.ctx.font=this.currentFont;}getFontSize(){return this.currentFontSize;}getBounds(ft){let Bt=this.ctx.measureText(ft);return {x:Bt.actualBoundingBoxLeft,y:Bt.actualBoundingBoxAscent,width:Bt.actualBoundingBoxRight,height:Bt.actualBoundingBoxAscent};}drawString(ft,Bt,dt){this.ctx.fillText(dt,ft,Bt);}drawCenteredString(ft,Bt,dt){this.ctx.textAlign="center",this.ctx.textBaseline="middle",this.ctx.fillText(dt,ft,Bt),this.ctx.textAlign="left",this.ctx.textBaseline="top";}setLineWidth(ft){this.currentLineWidth=ft,this.ctx.lineWidth=ft;}fillRectangle(ft,Bt,dt,It){this.ctx.fillRect(ft,Bt,dt,It);}fillCircle(ft,Bt,dt){let It=dt/2;this.ctx.beginPath(),this.ctx.arc(ft+It,Bt+It,It,0,2*Math.PI),this.ctx.fill();}drawLine(ft,Bt,dt,It){this.ctx.beginPath(),this.ctx.moveTo(ft,Bt),this.ctx.lineTo(dt,It),this.ctx.stroke();}drawPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let Bt=1;Bt<ft.getSize();Bt++)this.ctx.lineTo(ft.getX(Bt),ft.getY(Bt));this.ctx.stroke();}drawRectangle(ft,Bt,dt,It){this.ctx.strokeRect(ft,Bt,dt,It);}fillPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let Bt=1;Bt<ft.getSize();Bt++)this.ctx.lineTo(ft.getX(Bt),ft.getY(Bt));this.ctx.fill();}drawImage(ft,Bt,dt,It,v1,Vt,e1,t1,g1){if(arguments.length!==9)throw new Error(`drawImage call with ${arguments.length} arguments unimplemented`);let c1=this.canvasCache.get(ft);if(!c1){c1=document.createElement("canvas");let p1=ft.imageData;c1.width=p1.width,c1.height=p1.height;let l1=c1.getContext("2d");l1.globalAlpha=0,l1.putImageData(p1,0,0),this.canvasCache.set(ft,c1);}this.ctx.drawImage(c1,Bt,dt,It,v1,Vt,e1,t1,g1);}isDarkBackground(){return  false;}};var OP=class{constructor(ft,Bt){this.canvasElement=ft,this.changeListener=Bt,this.drawContext=new $n(this.canvasElement.getContext("2d"));}getBackgroundRGB(){return 16777215;}getCanvasWidth(){return this.canvasElement.width;}getCanvasHeight(){return this.canvasElement.height;}getDrawContext(){return this.drawContext;}onChange(ft,Bt){this.changeListener?.({what:ft,isUserEvent:Bt});}getClipboardHandler(){return new IP();}};var f2=`
/* We can customize editor styles here. */
`,C8;function bu(){if(C8)return C8;let j=new CSSStyleSheet();return j.replaceSync(f2),C8=j,j;}function H8(j,ft,Bt){let dt=-1;function It(Vt,e1,t1=0){e1.button>0||ft.fireMouseEvent(Vt,e1.button+1,t1,Math.round(e1.offsetX*_e),Math.round(e1.offsetY*_e),e1.shiftKey,e1.ctrlKey,e1.altKey,e1.button===2);}j.addEventListener("pointerdown",Vt=>{dt===-1&&(dt=Vt.pointerId,It(Bt.MOUSE_EVENT_PRESSED,Vt));});function v1(Vt){dt===Vt.pointerId&&(dt=-1,It(Bt.MOUSE_EVENT_RELEASED,Vt));}return document.addEventListener("pointerup",v1),j.addEventListener("click",Vt=>{It(Bt.MOUSE_EVENT_CLICKED,Vt,Vt.detail);}),j.addEventListener("pointerenter",Vt=>{It(Bt.MOUSE_EVENT_ENTERED,Vt);}),j.addEventListener("pointerleave",Vt=>{It(Bt.MOUSE_EVENT_EXITED,Vt);}),j.addEventListener("pointermove",Vt=>{dt!==-1?dt===Vt.pointerId&&It(Bt.MOUSE_EVENT_DRAGGED,Vt):It(Bt.MOUSE_EVENT_MOVED,Vt);}),()=>{document.removeEventListener("pointerup",v1);};}function cu(j,ft,Bt,dt,It){let v1=typeof navigator<"u"&&navigator.platform==="MacIntel",Vt=t1=>v1&&t1.metaKey||!v1&&t1.ctrlKey;function e1(t1,g1){let c1=D2(g1,dt);c1!==null&&Bt.fireKeyEvent(t1,c1,g1.altKey,g1.ctrlKey,g1.shiftKey,Vt(g1));}return ft.addEventListener("keydown",t1=>{Vt(t1)&&t1.key==="c"||Vt(t1)&&t1.key==="v"||e1(dt.KEY_EVENT_PRESSED,t1);}),ft.addEventListener("keyup",t1=>{e1(dt.KEY_EVENT_RELEASED,t1);}),j.addEventListener("paste",t1=>{let g1=t1.clipboardData.getData("text"),c1=It.fromText(g1);c1&&c1.getAllAtoms()>0&&Bt.addPastedOrDropped(c1);}),()=>{};}function D2(j,ft){switch(j.key){case "Control":return ft.KEY_CTRL;case "Alt":return ft.KEY_ALT;case "Shift":return ft.KEY_SHIFT;case "Delete":case "Backspace":return ft.KEY_DELETE;case "F1":return ft.KEY_HELP;case "Escape":return ft.KEY_ESCAPE;case "Enter":return ft.KEY_ENTER;default:return j.key.length===1?j.key.codePointAt(0):null;}}var XP=class{constructor(ft){this.canvasElement=ft,this.drawContext=new $n(this.canvasElement.getContext("2d"));}setDimensions(ft,Bt){this.canvasElement.width=ft,this.canvasElement.style.width=`${ft/_e}px`,this.canvasElement.height=Bt,this.canvasElement.style.height=`${Bt/_e}px`;}getDrawContext(){return this.drawContext;}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}};var wu={"chain.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB2UlEQVR4XmNgGJmAkYGJgQULZGXAhCygIGICQg4GdjTIARRDh5xAEQwNKYzXeK4xY1Oez3SNfz8TmoYUxq+6m9bfqLjDI4tmfhzLff9VR17FoWhwZXpkOm2v1C+79+cbD4gga8lneuhrfdv6/+ItSBpSGB+bTDos9IfzP9N/k3cX6+7wwByWz/TWR/E1w/+kO+8j4BpSGL9rz9wh+Efs14TF3heY/tu/vlF2hxtkSxzLY2/FVyz/om69jT0G8kM6QyNDM9MT41nbZX9y/J/ct1/4iU7aIYb/9q8uVx0UbmF64mV7i/1/0s0HSVM5dRgYrjB84/8U+bh80iH+34z/17Xc51ZhUGX8qpkK1GL6+krltyClVyz/c+69C57CwQsK1v+Sl9rdvgr8Z/7P9WvKxNs8fsD44ABqeWLkdYHhv9sL8e+s/xJuv4k4yajEAI6H/+6llxn+g6DUh75gSXhgugEdmXIUJB5150HSFA59oHKIBteGSyBhpv+q9wsVQaZDICdDKuNnneRjCY9exU3l5AUrh2gQelAQ/8Tgve2Tb25NjIgYBkmnMl5Sumi+hQmiGKqhgaGB57Pts4Bfek0sjkjmIxQhs8BpiZFBh0EPCLGlTnQxcGod8gAAp9es6fCW8G8AAAAASUVORK5CYII=)","eraser.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAC1UlEQVR4XnWU309SYRjHn8OYTTM3zTmjC8NFFBR10dJu27wx11VX/QNdeGebTsvua93mRbUmLhURzYgKRZRJUJLibLPpRDCZC3/b1AMc4Tw97+EAR2e8FzD4fr7v53nPOQDCDkT/s2KAKiwROOXvFP8JLnDDGHhgArzgg28wCQGYgiDMw6E+1pY2rsECLEIIIrAC1O2CV/AWzPAOLGCFQbDDJ/hC+CQktU5bXWJoMGlaVv0iPAy/GeCmeBf0QB/YYECKO6kkzAl6Z/+t+BmsSJotgmmN4lFYZcA4tfdQez+1D8Fnio+S1oHe5TAcFuJHVBNit4nGLYrHGOCBbin+ntodMEzxKMkMWy8LpfgSEUN4niC7TTBtqzYYMCG3Z9xHadgD/bC1hi+U4uw1TkhFsqs3adpngJdGHSAZB8VHYAZ4vdt+RSjED3KcvYWYWMJuFY0E+MjdQe5MZgYE7YhFL8soCJzDU1iKwdcE+HOjBoG/NNJbq5DJI40EPNiIPCLgO/Wzk4lS3D2kPyaTQe5hETZsLDfxRRQLkLuLDjJd7u8wJs/mRs22J7CR4re3Qq3xIpENPUUHG+biuqX2UVsVX41epTp9ZjINm0ttqTIEZECQboK0ztd/Ya951t+nRi2GFYgksxluSlO7BMTYLVblMVcmOOTEZ9N+i4YvkREmAySz2JaNE4CqtM5jvrZHcbbE5lmLU8NrJTEmc3drsT1VlmmXdsCbgTdSu7wKUi2zPqsaq7GWZCjeJBZnwxlAN9dRyecBtsvzH/4+DV+MNdsLT4/GJSU0jHde3T2KtMxYnA8X5tvT5cp2eehDLnVxrPPoLgWpx4Hd+8Lp/Zx7boYoe/g40eDtViIF6Se+xPX8qPl96DqECImrRMOY2fg3e1atwd16UX1cR75wEXr4VmGdxL52atbpX2HnxfTenfiJcelKr9CjvQp/IMWJ57AO62ndOKk7890/iSWBU4XaZc0AAAAASUVORK5CYII=)","fist.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAABj0lEQVR4Xu2S3yuDYRTHz7IZsfdCUrYLG4VCpJV6d0mS/A1zg79ArRArP4q5s4ulVousFCn/gX/g5WK7wkZMzIXMxjZ793XeMdn7vpM/wDn1PD1P53PO95znIfq3rwmYyU5tZPr7PJYoTjGaZ8xB1t+xZQqyP60iAEeWsThFGKpqbiE9CRECrnCHLnTAj17QbLV4E3mFXALbcOAGSXRiClnMgZLkZR/QYg0UEpDBDuxI4IHzT6OADdRiEK2gE12gBWmWE8Yj7weQIGOT650zaoiqAVtjDx2KeEWR8xZ4lUt+jEWkWJxJBYgkWa7NLy5WXWl5FilzhfoKwEAzzkwYI3DhTQUoVYpYR10FYCTPaDbJ2heQUwGfR0kFELmt6Utu9Jmz6dmFpmmR7v141w1XBiCBNFNacyKmCxT59kgHGBIQ4g7UkpRzESuwaCoQ+YZxqwGUGaUwrvvStqazLeRLj/bTZUTQB5rQfg1jjac9e8qAzFgZKbDMALpj1Kz3Z0WKjiGIfex++x6H98PgK4d/APiDpdwStyBoAAAAAElFTkSuQmCC)","hand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB50lEQVR4Xq2TwUsUcRzF3yy52TboHiKVLBCElb1479g5kgKPCkKs4ME6eAiiP0GELileStew6NLJq5Sihw6BG2VlECymB2PVXd10Zl7vN+OuOjNLl35fZhiY+fzee9/vb4D/tAow9fafu90BsqrvnXzNBwSTTBAZ2I3BkSbeYBMzLHNRn9/nEFHCvYZAutDHDfb5wDvanOVnoozBhkBzIUeHOQEHfO8DW0QV0xjC3VjoUh2onAC/ZPAme4giemMQo+BKoaeusCmdSa41MmYL8HxLgUKe27q/UKLzSSxcwUOMwk7VgTKX6sBz7oeA61hJHSUP8ST1raZQA7Z8hb0QkO1WE+d5lZdlx+UbTsjSR95i0bc0E7GUzfIHdziouRqFPzwU5ii4IzOP+SFiqe3i8jyP1A1bgKPyVGa5/M2S7iaDdWZ8KUwNe1Uus923FHx8ujwZPA9YGO48+CK/Ywp4un8N8SIZgG78fCTvFV1n9w+ePe6GFMzMJ7u4zuOInQDYYaKcDB3Ba63FGQWPLk8mt2MAWKv96kk0sKM2TBFfkQ4fvkx6fyHSH9NalwO0PkXPaktirt8rhRpqelblbV6IASzkmivP9NrEdE9AM/FXbCPG4v+6p+3VcZ2ql5zTZSqv+XcQ6+gKgL8a/RiIJPdySgAAAABJRU5ErkJggg==)","handPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACAUlEQVR4Xq2Tz0sUcRjGn1lytW2wPUQamSAIK3vp7kk6i1LgMUGIFTqUBw9BiPsPhNAlpUtuKhpdOnVUKqlDh6CNsh+CsJQdjC13ddOZeXq+M7uTzu7SpXmZLwvzfr7P8/5Y4D89eZh48s/bhoC04nMXH/EmwThjRAp2c/B6C7vZwhRLXFP6NY4RRVxpCiTzw/zCYR94RpsP+Z4oYbQp0JbP0GFGwB6f+8A2UcF9jOFyQ+hkCJSrwDcZ7GcfUcDFBohRcKXQFyp8lc4s3zYzZgvwfEuBwgK/65xXRccrsXAGE7gBOxECJb4IgQfcjQAX8DJxEN/HVOJTTaEGbPsKvyJAuldNXOZZnpIdl485I0tveIkF31KuzlI6zU3ucFRzNQq/uS/MUeGOzNzm6zpLHa3ryzxQN2wBjsIT4JE6f7Co09RgHRlfAnPjXoXr7PQtmdSsNqn2eDJ4HLAw3rX3QX4nVaC534sA0bYCvdi6Je9lveb+mkLw2+PPiIKZ+WwPN3ioj9MyczQMsMNYKR5ZwfOnCzkVTq7KTpYDgrKCp5XuyGw9AOvViHoSmPhryaS7nCM+IhldvlRy92nYmaBLQWtdXqX1rn5X22OLI16xihhjBjA9q3CQJxoAFjJt5Xv6bBLdKmgmvsIOYrLxv+5uZ+WOtmqJi3pNLGj+54gN9ATAH0reBFwT+FQbAAAAAElFTkSuQmCC)","invisible.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACjUlEQVR4XnWTS09TURSF178wMUExvhIfiY+0Q6MThwwEHYk6MODAqHHgzD9gHDgWYhQBAUWMqY/SogSkCGJJNAiFUiyV2qqURkFa7cPvnJZiBXMH7Tl3rbXX2ntf5bWgyH+emPI8v8reAn8nj3r0Qr3qU78GNKghDWtEfk3ot2LKKq6AphTUjGYF26ObuqU7alabOvRQj/VEz6APKS23jvI/rY/6AD2ksCH0AG9Sq9r1QJ0W/hyREFbcOqQNcqiL/3FuIpozhJeot6J+H/VHegrci60lSA7t0VVtlRNqTvPAY4bQqxYL70LdxSsvd2l164B2qVHLVN4L0U2VhL4aQl9RveDdS9gl4IcBNWpRUShNUJxUT3OG0E/UTo4u4N0a1U9SHdRu3QOQIOw8N+0Yc/A2ZwgDeHfh3ZgZpbBH+zDTAPy7Jok6qU/K0Jht2k91CL5SVD9aHh1BvYHiCxYe1nV9w9h5bVcdk4DwGn3TmYg1Y9Rb8P2FQYVQbOM+qjPaDDwMguMw3j00Mst0narUDaUgBICPI5VgAhe5rUVguZBhhMaGOEzrrU6pQpfpdoqeB8k2hvoF7UB9mhxmsyD4WYIs0c/ifEKnoVxSEsptQke5rdQ5vGdRt4SYXTEf2lWq1l1s1GojlDiAgpmTLN4KHEKegw+NKvsco8oglApd0XtoOzEzhZmCuq2QJ7RRX3lqmO84xjaxGltUj/dVcJEwhuIqwVRptsZMZwL/wG2FPIHryijVVBnUNdJlS1ZKlmJEzkApr1JDj5KsyeJaQsR+fDlW8G/KCWyl1tgpziEIxczwVcnYcT6p5LpwO7gZ5jzH7mT0hmbW0+IO/UBixXX5L4RZlmpOn8u6vT7Y3P4BKLGmkBK1qvgAAAAASUVORK5CYII=)","lasso.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADg0lEQVR4Xk2UXWxTZRjHn7XYfWhh4DKlIFLRdlucImbGr2gEwYwsAjGRXXihRlTOFrOoF0RcdWwhEVnQdgOmbgzoliEhAZJlvZwdMQg6No/U2IYOWVrK1q1d19q1Pef8fU7bsy3/nIvznv/ved/n47xEZKDivNpK8RoE6XC4J9gX7A/2hXukwxCwta1UcxhUu4PcLFHnqZnaf+/E0FhTqDpiiaqqjjSFhsbvnZja76kRdarLQRzbTSjAuozgczaHVoJYhShX1krlSmH2bSWaQz5nRsA6MJQDrHMtx0Y3y4QS7Eo0eTpG3K5rP7tdHSNNnl2JNQxtlo+NzrXAmgXEDcnWA/5ihbAtOeyKtKf3wmIvs6+2l8GS3hv51u3aliQUKwf8yVZxA2GNIvSIBCNaJmPt/jrR6KZfyL6omQdQF2tvmTTyPt2iIhDq/f1mth/1LtjCVTa9QAIN8dEl8tK8mgJBj6oF21GvEWb4+wlft08Sds+FHbPmDl2A/mOl2Ran7SQylkV0MIcdu+cI7J3utUb0aL2q1DZSlDiRvObpKdpCZymVf1dqW6/qYY1QsM8SXYULw3jcq8XLWlSghOrJpYUwXxpeDUuUZn6smNWheUx6l7uxGD8HrKXz9A2doQidJnnPoWt6VEQItm4/4aX0LSd2wLCEqMB6+otmqYt8K1LVd3+oTRA6/yXUTXe+mCHsjE048QbWw8R6EHoNAMlGbA+d/PSOAZXKZC9tKbr19m+DFZztc/E/B2/3jp//x5logyC/8oyBd9DBIjeODj4/X8hTcO4PZZ/a6RKpYfBKWb6CuTQ+i41elD5A5cLHga7jPjOvWDIDv6e/EE25WXo49N0mPmF21OR3wm/FTahMf3/dNtLoe4GPa1LeC9y4xGUxacNnvHnykfTG+IWxJzOlyhnPzHH7bUJRvgINiSuu+CE8i2Jw8NwORq/jidRDktMz4COU46twnVyEx+brwxfFQDc+weu4P0drQFHq/Z+uP73UZq7vh8FAV6oeL2OTZl4OFMCUaZw4NXDjyNTBWENylaLDwXFpH+5Tm5mku4sN1XZQFwrkR+OvyjvxpvT5kTuEHcmpDh5i/qIQ/1l5LQd4RifUOS1A1XTnRq76r5dhzRlTPBxx7vlSDstmiA0r8NGXUcLZm6jJARkGZvnJAtqtod4JmhJbT/9tiZ4bS9QsX83dGv8DIrFVpnd+f4UAAAAASUVORK5CYII=)","lassoPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADiklEQVR4XlVUbWxTVRh+12K3oYWBy9SCSAHbbXGRaGb8SoygmJFFICayH/5AIyrdYhb1BxFXHWswQRa03WVM3RzYjaEhYcZl/Tk6QhB0bN62xjV0yNJStm7tutbetvfex3N7e3XmyUnuOXme9+O8zz1EZKDyIhwVeBE28VisLzIQGYwMxPrEY7Bhh6NCYxgUuou8DLwuUD936G736GRrtC5uSSioi7dGR6fuds8dCtTzOoXlIhbbSyjBhrwt6G6LrgExlKJKfkiskksLuzVoiwbdeRs2gIlUgXWp/eTEdomwGnvSrYGuca/n2g9eT9d4a2BPej0TbZdOTiy1w1oQ8JsyHYdD5TJhZ2bME+/M7YfFWelc56yEJbc//oXXszNDKJcPhzId/CbCetnWxxOMaJ9NdoYaeaOXLpHg8HOCw0lOWrgPjcnO9lkjy9PLyzZCU2jQzOgnpgV7rNaut5GNRsnPEXzcstICQY9awX5i2ggzQoOEzzpnCXuXYq5Fc5cuTH8z5IoCnkRVooM55tq7RGDc+X5rXI+Oq3JDCyVILhTj54bPEYbPKV+CoyAhuaHjqh7WOEUGLIm1uDCGbdOFeEoxK+Hjilvz8Ng6WBK08E31og5tk+IBNg3GFBw+zlfMoHwJjrMUpzMk7Tt6TY/qOMHeGyI8l7vpxi4YtOBq0xvJR4vUQ8FV2bo7XzekCdxfhMZ57tk8YXdyxo1XsBEmhvv93ZoAJBnxcvT0B7cNqJFn++mJspuv/zJSzcb2VOr3kVv9Uz/+6U47BI/vZ+FzlkEHi9QyMfL0cilzwfnf5IPKpFeLzSOXK4s3qDb8YXLiovg2aoT3wj2ngmZ2YskP/Zr7mDepXnow+uVWVmHBatIbsddSJtTkvrpuH28JPsPKNclvhm8Ms2sxaeYz+k8/nNucujD5WL5CPhtYOOW8RSgr3kBz+rIndRRPohwsuJrBOO16NPuA6A4MBQlV+DTWKJVhy3JT7CIf7sX7eAn3qmpNUJZ969vrj7PGNRjwTiTck23C89iqkVcKSmDKt8x8N3Tj+NyRZHNmrazDkSnxIO5RhpmhO/9G0jIoByXSI6kXpN14Vfzo+G3CrsxcFzOx4iJif1YRKwXMozNssZi189xmdutXfoJVJWaZOVJs5v/18H+/rcK7nyQI3/tRrwryTLDIVkGgvRrKm6AhvePMH5bE+cl0/cpT9dX4B08ibwJFFp9KAAAAAElFTkSuQmCC)","pointingHand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAADe0lEQVR4Xq2V+yvrYRzHnWwuaaUst9mY21iyHcVySghxJJLLDy7L2Y9S4geHThx+5ISc0vCDf0AS6fj1rPxkRJ2dJjvtjNolt1y+NMY+5/N5yonZbIunvm179nyf1/N+fy5PSMgLo6OjwxAWFgaxsbHQ2NhonJiYELy0Pqj/CgoKPqSmpjoGBgZgfHwciouLna2tre+D2uSlxXK5XJ2UlMTt7e2B2WyGjIwMiImJ+fhmADy9OjMzk7PZbHB0dAT4HXp6en6+GSA5OVktk8k4u90Ox8fHTEF3d/fvNwNIpVKm4AFACgYHB//Mzc0pUZ3o1SBPgEKhgKysLFd5ebkdlekTExMLUKUU56VVVVVhQQM9AdnZ2dDe3g5bW1tQU1MDKSkpNrTtr0qlsvT3939/NSA9PR2mp6fh7u4OhoaGWG2gXdDZ2Qnx8fFWBH3CR4P1IvMFe4feqoqKir7k5OQoSQEF2Wq1siwiwMzMDLjdbhgeHoa8vDw4OztjiiIiIqCyshLq6uoA39mRSCSKZ5DQ0FAxnuQXeutG+Yb8/Hwt1gIDHB4eMsDs7Czc399TsAFPCxzHwe7uLojFYlhdXQWDwQBKpZLDfdTeVMhra2vBaDRCV1cX2zAtLQ0IcHp6yuYWFxeZRfPz89DX18cAVIgYbNje3oaTkxPAJODwsN4BLS0tTPbGxgagTBCJRECFdnt7C+fn53B9fc0UXF1dweXlJbhcLnA4HAxmsVhYvVRUVHA8Hu85IDIyUl5SUgImk4mduKmp6T+ANiXvHz+Ag+ZJEYHpkxTgHt4VCIVCCTY009TUFNzc3MDk5CTlPRwcHLCN/A2CE6C0tNQ7IDw8nI8Z8LWhoeGWFGxubkJvby+THSyAz+d7jUEIpp46ISGB0+l0zHeKh9PpDAhACimd0QUuKirKO4BSC3uOTqPRwMXFBQvig/+BWER9C+8RLjo62jcAPayhvqPX6xmAgkf++hu0huKF2cdhivsGjIyMCAsLC81jY2PMHgIEMgiwvr4OcXFxHNaCbwDZ1NbW9q2srAz29/fZ6QOB0LrR0VHAdPd/ZwgEAhF6uaPVap/EwbMOHpTRPGUbKgfsBv5vPawJXn19/WfsR04qPMoozzg8hlEirK2tMUBzc3Pg93Zubq6uurqa9aCVlZUnz/LyMvu9tLQECwsLgPVD1f/Ds8n9AyubuM4FiwqZAAAAAElFTkSuQmCC)","rect.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACc0lEQVR4Xo2UT0hUURTGf7uipMy8GbUwigx0WbiwpCgQKmgl1CYj3Plahtg/eguFUIoamNxUCEFCGlIgEhSORmQRNeVMSkhpWcSU1KSOQ9i8vvfmpiMZOB8Xzjv3fPPd8+dertPHQICX/OAVnTRTz1GOaNXL7pQvSdTG9KHwCVJMyzlMFw1UsJFC8lmtVSi7Qr4u7SUVk1KsmCkyZtJETchUmTKh3NSYk6ZBq0a276nSXtRMGc+kfMIkiRY3shQkWiZ9wjjxMN5SEAuPQ5qegBALe07a+eiMCj9lxZ0ZJ+N483A9oQf6qbWED6rITW4IcQGaGJuT/Y0f7rm1cJliS+jnLjNkAviEDXTwVbZ/2gwjobLESKgY6iiwhI5gyw/N/T2zKtljF0CN6p39SAZbCVyhTpwVLKOUM8wGmn4lEy35yGH+Jp1TKF9nrRp3mzbOMcp9m0MhtLPrvwRDr/6/mwda6cZ4ON24Hd5z1RISfGNIAdmDnWabtHtlzxLjtY1pgucM2o8rPOUhj+UZY0qDWGEJad5y0ca8mO90LLyKA+yjkirN6QWeaO0kwjvucZw8m0MwGinLzmelUi1nP3vYQrUCH3GNY5Qp+UpbpelcQjNnWcchVWRQhc2jiPWsYTl7aZRetvTBtP5VaJW7la0c5g4npFbCbk5xSzlNKPGRUKk6vYCwScIH2ax2lagH1VJ6wxdbNVUumKUFhGFNaMxiyEk4v3Jn1VmE4LluBK804W/5zdDA/+OxCtk58dzeNjcS7vYJ/v1bzGMJnm7rPDLmkzlvdph2E5/zRs13M2BS/p3OvhoLkVSBi7jE5zn/VPCy+K/GH86K+gV5WsL6AAAAAElFTkSuQmCC)","rectPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACi0lEQVR4Xo2UT0hUQRzHP6c2KmrTnYoKjCAP+45FB0uKAqECT0FdMsKbr2OIUdE76EUpamHzUmoEGWmIgUnQn9UOWURp7ksJKSuLeCW1qesStq/fvB11lzr4vgzM/Ob3md+b3/xmuEYfA4Fe8ZMhOmmklmMclVYr/U6xpRg0Pn2I+yRpZsQ4Shd1lLGJCGHWSItIv0xsXTKXEp+0+AqZJqum1KCKqQpliXaqKnVK1Umrkr62VMjcoJpWvkprYAqvyUksRV7TlAYmcOP4S1EyPgEZegMgGfftjP3JHhf9kp5rz9pZ21+U44t6oZ9qA3yUjNygReSKoIEPC2H/oN19pxouUWKAfrqZJRtIAxvp4Jv09d9mGYtZ3lisBGooMkBHMKVd5QsRlSbfcxMl99tFUCX5zg1SwZSHI6qJ4q+wNHWGuSCmzqTXFEYMan7TeYlyBSi2ItymjXOMc9/sIQLt7C4AMiE36ka7K/HDlUXR1uh0qIcH9JCpd+OZ+u3wnisG8PjOCEOydr6S1hxJXhufBnjBsBlc5hkPeRS6bt2zbuUiWK1WJpThLReMz8vFk07GV3OQ/ZRTIXWqN11t7SLBO+5yglVmD0FppA0dZqWkcScH2MtmAVqsJ1zlOJZUbbnJ0kw+0MhZ1lEpGRnGCS2zVGgDa1nOPup5ahYNqnU+QrOYm9nGEe5wUqKVsofT3JRdTspZjMWictIFwBYJfIitcsClFHNYIr3hq7iapAW1VACMSoUmjUZsz/6dX6t2rvgKAN9xEvhRT0/oZaXg/7EYIFcnvvO4zUnEezSg79//LAbw5bYuKqs+q/Nqh2pX7oJ1UP1QAyqt73Tu1ShUShK8not8WbBPBy+LfjX+AjC0Avv8MQHTAAAAAElFTkSuQmCC)","zoom.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACiUlEQVR4XpVTTUgUYRh+nJWCxUsNevBiYGm7UjAnFyoI9ORKSxt4a71Ei9I10EPg0VO7hOE1ikQSofXiYv6QRX9bkTAjbbM/B2mz8id3Z2f/ZvbtndR0NYn9XuYbPuZ55n3f730efMQCZjCLOX4/xyJe4BXe4C3e4QNi0JDCBu8JRPkURxIMm8BjjOMJJvEUU8jadUn36D79Zs5X8JhSyb6AX/iBzxxfLMIMwx/gIcYwXRMTNa86Mhrp23QXOwx3sW9zNKKOaF5NzAlrnCWBFXAx43jE8PeC0b4U9KstJmh/tJh+dSlotJeEDYZ/Bea5GAtudoZDkm5jcCP1poejd+XhaG+6kc82kvRwiDpLwha+A1RX8K4OaDfCU038UTQGk/GA2qO4lDbFpfbEA4NJ0QA1UThktOeFDBPOLd+/qF9PtOZBDi30LNU91FBfgz+rvmaoIdUVmnZo4CxLQU3cYsLJ7G1HebvqC7nIhHJ6G7y7mmsNKTQtGjbyq5p3Dnzv2au31i24QK35cEhpriQAzbUp92AS1GKqI7odL5F1jM9bzV7KzoyVLvfbDhKAoYZ4wGp/NFKU8Br6scS9VrM7Fw8a56/8A271ovb0pkF9m2UPy2AWGd/k4sYdpal/7+fH4cTxvaPsGo6C3EXysWYWQafIR2cqSnFyV237CM6ADOowyM8Si+xkkCsztB2RIYZCFT2QB1mYVdwSSSza/N85nD1qDl27cyA76EQ1k6bqtESiRagrXFsdyBxQq+ySnfIhtbJ6gHX2bBprQvmAHwKH/GDBmRCDyibP4Od/HLcNZ0KcCSnkmLQsKEd4mmvfgTMhybGCb0zI4RPb1bSbUtnDUvHz4yGJ7BUWx2+5Iu7Rc33AhQAAAABJRU5ErkJggg==)"};var M8=Object.create(null),gu=3/4,GP=class{constructor(ft){this.HOTSPOT_32=ft.HOTSPOT_32,this.IMAGE_NAME_32=ft.IMAGE_NAME_32,this.cPointerCursor=ft.cPointerCursor,this.cTextCursor=ft.cTextCursor;}getCursor(ft){if(M8[ft])return M8[ft];if(this.IMAGE_NAME_32[ft])return this.buildCursor(ft);switch(ft){case this.cPointerCursor:return "default";case this.cTextCursor:return "text";default:throw new Error(`Unknown cursor: ${ft}`);}}buildCursor(ft){let Bt=this.IMAGE_NAME_32[ft],It=`${wu[Bt]} ${this.HOTSPOT_32[ft*2]*gu} ${this.HOTSPOT_32[ft*2+1]*gu}, default`;return M8[ft]=It,It;}};var KP=class{constructor(ft,Bt){this.title=ft,this.rootElement=Bt,this.elements=[],this.dialogElement=null;}setLayout(ft,Bt){this.hLayout=zu(ft),this.vLayout=zu(Bt);}add(ft,Bt,dt,It,v1){this.elements.push({component:ft,x:Bt,y:dt,x2:It,y2:v1});}createTextField(ft,Bt){return new q8(ft,Bt);}createLabel(ft){return new S8(ft);}createComboBox(){return new V8();}createCheckBox(ft){return new E8(ft);}setEventConsumer(ft){this.consumer=ft;}showMessage(ft){window.alert(ft);}showDialog(){let ft=document.createElement("dialog"),Bt=this.rootElement.getBoundingClientRect();Object.assign(ft.style,{position:"absolute",marginBlock:0,left:`${Bt.left}px`,right:`${document.body.parentElement.clientWidth-Bt.right}px`,top:`${this.rootElement.offsetTop+30}px`}),this.dialogElement=ft,this.rootElement.getRootNode().append(ft);let dt=document.createElement("div");dt.style.display="grid",dt.style.gridTemplateColumns=this.hLayout,dt.style.gridTemplateRows=this.vLayout,ft.append(dt);for(let{component:e1,x:t1,y:g1,x2:c1,y2:p1}of this.elements){let l1=document.createElement("div");c1===void 0?(l1.style.gridColumn=`${t1+1} / ${t1+2}`,l1.style.gridRow=`${g1+1} / ${g1+2}`):(l1.style.gridColumn=`${t1+1} / ${c1+2}`,l1.style.gridRow=`${g1+1} / ${p1+2}`),l1.append(e1.getElement()),dt.append(l1);}let It=document.createElement("div");It.style.display="flex",It.style.flexDirection="row-reverse",It.style.gap="15px";let v1=document.createElement("button");v1.textContent="OK",v1.addEventListener("click",()=>{this.consumer.fireOk();}),It.append(v1);let Vt=document.createElement("button");Vt.textContent="Cancel",Vt.addEventListener("click",()=>{this.consumer.fireCancel();}),It.append(Vt),ft.append(It),ft.showModal(),ft.addEventListener("cancel",()=>{this.consumer.fireCancel();});}disposeDialog(){this.dialogElement!==null&&(this.dialogElement.remove(),this.dialogElement=null);}},an=class{setEventHandler(ft){this.eventHandler=ft;}fireEvent(ft,Bt){this.eventHandler(ft,Bt);}},S8=class extends an{constructor(ft){super(),this.element=document.createElement("label"),this.setText(ft);}setText(ft){this.element.textContent=ft;}getElement(){return this.element;}},q8=class extends an{constructor(){super(),this.element=document.createElement("input"),this.element.type="text";}setText(ft){this.element.value=ft;}getText(){return this.element.value;}getElement(){return this.element;}},V8=class extends an{constructor(){super(),this.element=document.createElement("select"),this.element.addEventListener("change",()=>{this.fireEvent(2,this.element.selectedIndex);});}setEnabled(ft){this.element.disabled=!ft;}addItem(ft){let Bt=document.createElement("option");Bt.textContent=ft,this.element.append(Bt);}getSelectedIndex(){return this.element.selectedIndex;}setSelectedIndex(ft){this.element.selectedIndex=ft;}setSelectedItem(ft){let Bt=this.element.options;for(let dt=0;dt<Bt.length;dt++)Bt[dt].textContent===ft&&(this.element.selectedIndex=dt);}getSelectedItem(){return this.element.options[this.element.selectedIndex].textContent;}removeAllItems(){this.element.innerHTML="";}getElement(){return this.element;}},E8=class extends an{constructor(ft){super();let Bt=document.createElement("label"),dt=document.createElement("input");dt.type="checkbox",dt.addEventListener("change",()=>{this.fireEvent(3,dt.checked?1:0);}),Bt.append(dt),Bt.append(ft),this.element=Bt,this.checkBox=dt;}setEnabled(ft){this.checkBox.disabled=!ft;}isSelected(){return this.checkBox.checked;}setSelected(ft){this.checkBox.checked=ft;}getElement(){return this.element;}};function zu(j){return j.map(ft=>ft>0?`${ft}px`:"auto").join(" ");}var qA=class{constructor(ft){this.imageData=ft,this.dataView=new DataView(ft.data.buffer);}getWidth(){return this.imageData.width;}getHeight(){return this.imageData.height;}getRGB(ft,Bt){let dt=this.dataView.getInt32((Bt*this.imageData.width+ft)*4,false);return (dt&255)<<24|dt>>>8;}setRGB(ft,Bt,dt){let It=dt>>>24&255,v1=dt<<8|It;this.dataView.setInt32((Bt*this.imageData.width+ft)*4,v1,false);}toDataURL(){let ft=document.createElement("canvas"),Bt=ft.getContext("2d");return ft.width=this.imageData.width,ft.height=this.imageData.height,Bt.putImageData(this.imageData,0,0),ft.toDataURL("image/png");}};var YP=class{constructor(ft,Bt,dt){this.canvasElement=ft,this.dialogRoot=Bt,this.JavaEditorArea=dt;}register(ft){this.javaUiHelper=ft,this.cursorManager=new GP(this.JavaEditorArea,ft);}grabFocus(){this.canvasElement.focus({preventScroll:true});}setCursor(ft){this.canvasElement.style.cursor=this.cursorManager.getCursor(ft);}showHelpDialog(){}createImage(ft,Bt){let dt=new ImageData(ft,Bt);return new qA(dt);}createImageFromBase64(ft,Bt,dt){dt=dt.replaceAll(/%\d+%/g,e1=>"A".repeat(Number(e1.slice(1,-1))));let It=Tu(dt),v1=new Uint8ClampedArray(It),Vt=new ImageData(v1,ft,Bt);return new qA(Vt);}createDialog(ft){return new KP(ft,this.dialogRoot);}runLater(ft){typeof requestAnimationFrame=="function"?requestAnimationFrame(ft):typeof setImmediate=="function"?setImmediate(ft):setTimeout(ft,0);}};function du(j,ft,Bt,dt,It,v1,Vt,e1){let{readOnly:t1=false,initialMode:g1="molecule",initialFragment:c1=false}=ft,p1=document.createElement("div");p1.dataset.openchemlibCanvasEditor="true",Object.assign(p1.style,{width:"100%",height:"100%",display:"flex",flexDirection:"row",alignItems:"start",backgroundColor:"white",touchAction:"none",userSelect:"none",webkitUserSelect:"none"});let l1=p1.attachShadow({mode:"open"});l1.adoptedStyleSheets=[bu()];let U0=null;t1||(U0=document.createElement("canvas"),l1.append(U0));let u0=document.createElement("div");Object.assign(u0.style,{width:"100%",height:"100%"}),l1.append(u0);let $1=document.createElement("canvas");$1.tabIndex=0,Object.assign($1.style,{outline:"none"}),u0.append($1),j.append(p1);let M0=new v1(new YP($1,u0,dt)),St=new dt(u2(g1,dt),new OP($1,Bt),M0);if(c1)if(g1==="molecule"){let r1=new Vt(0,0);r1.setFragment(true),St.setMolecule(r1);}else {let r1=e1.create();r1.setFragment(true),St.setReaction(r1);}M0.setEditorArea(St);let He=t1?null:new It(St,new XP(U0),M0);function re(r1){$1.style.width=`${r1.width}px`,$1.width=Math.floor(r1.width*_e),$1.style.height=`${r1.height}px`,$1.height=Math.floor(r1.height*_e),r1.width>0&&r1.height>0&&St.repaint();}let VA=u0.getBoundingClientRect();re(VA);let C1=new ResizeObserver(([r1])=>{re(r1.contentRect);});C1.observe(u0);let B0=null,vn=null,st=null;t1||(B0=H8($1,St,dt),vn=cu(u0,$1,St,dt,Vt),st=H8(U0,He,dt));function q1(){p1.remove(),C1.disconnect(),B0?.(),vn?.(),st?.();}return {editorArea:St,toolbar:He,uiHelper:M0,destroy:q1};}function u2(j,ft){switch(j){case "molecule":return 0;case "reaction":return ft.MODE_REACTION|ft.MODE_MULTIPLE_FRAGMENTS;default:throw new Error(`Invalid initial mode: ${j}`);}}function ju(j,ft,Bt,dt,It){class v1{#t;#e;#P;#A;#f;#r;constructor(e1,t1={}){let{editorArea:g1,toolbar:c1,uiHelper:p1,destroy:l1}=du(e1,t1,U0=>this.#i(U0),j,ft,Bt,dt,It);this.#t=g1,this.#e=c1,this.#P=p1,this.#A=null,this.#f={[j.EDITOR_EVENT_MOLECULE_CHANGED]:"molecule",[j.EDITOR_EVENT_SELECTION_CHANGED]:"selection",[j.EDITOR_EVENT_HIGHLIGHT_ATOM_CHANGED]:"highlight-atom",[j.EDITOR_EVENT_HIGHLIGHT_BOND_CHANGED]:"highlight-bond"},this.#r=l1;}getMode(){return this.#n(),this.#t.getMode()&j.MODE_REACTION!==0?"reaction":"molecule";}setMolecule(e1){this.#n(),this.#t.setMolecule(e1);}getMolecule(){return this.#n(),this.#t.getMolecule();}setReaction(e1){this.#n(),this.#t.setReaction(e1);}getReaction(){return this.#n(),this.#t.getReaction();}setOnChangeListener(e1){this.#n(),this.#A=e1;}removeOnChangeListener(){this.#n(),this.#A=null;}clearAll(){this.#n(),this.#t.clearAll();}destroy(){this.#n(),this.#r(),this.#t=null,this.#e=null,this.#P=null,this.#A=null,this.#r=null;}get isDestroyed(){return !this.#t;}moleculeChanged(){this.#n(),this.#t.moleculeChanged();}#n(){if(this.isDestroyed)throw new Error("CanvasEditor has been destroyed");}#i(e1){if(!this.#A)return;let{what:t1,isUserEvent:g1}=e1;this.#A({type:this.#f[t1],isUserEvent:g1});}}return v1;}function Bu(j,ft,Bt,dt){class It extends HTMLElement{static MODE=Object.freeze(Object.create({MOLECULE:"molecule",REACTION:"reaction"}));static observedAttributes=Object.freeze(["idcode","fragment","mode","readonly"]);#t={idcode:"",fragment:false,mode:It.MODE_MOLECULE,readonly:false};get idcode(){return this.#t.idcode;}set idcode(Vt){this.#t.idcode=String(Vt),this.setAttribute("idcode",this.#t.idcode);}get fragment(){return this.#t.fragment;}set fragment(Vt){this.#t.fragment=!!Vt,this.#t.fragment?this.setAttribute("fragment",""):this.removeAttribute("fragment");}get mode(){return this.#t.mode;}set mode(Vt){this.#t.mode=String(Vt),this.setAttribute("mode",this.#t.mode);}get readonly(){return this.#t.readonly;}set readonly(Vt){this.#t.readonly=!!Vt,this.#t.readonly?this.setAttribute("readonly",""):this.removeAttribute("readonly");}setMolecule(Vt){this.fragment=Vt.isFragment(),this.idcode=`${Vt.getIDCode()} ${Vt.getIDCoordinates()}`,this.#e.setMolecule(Vt);}getMolecule(){return this.#e.getMolecule();}setReaction(Vt){this.fragment=Vt.isFragment(),this.idcode=dt.encode(Vt,{keepAbsoluteCoordinates:true,mode:dt.INCLUDE_MAPPING|dt.INCLUDE_COORDS|dt.RETAIN_REACTANT_AND_PRODUCT_ORDER})??"",this.#e.setReaction(Vt);}getReaction(){return this.#e.getReaction();}clearAll(){this.#e.clearAll(),this.idcode="";}moleculeChanged(){this.#e.moleculeChanged();}#e;#P(){this.#e||(this.#e=new j(this,{readOnly:this.readonly,initialMode:this.mode}),this.#e.setOnChangeListener(this.#l),requestIdleCallback(()=>this.#A()));}#A(){switch(this.mode){case It.MODE.MOLECULE:return this.#r();case It.MODE.REACTION:return this.#n();default:throw new Error(`Mode ${this.mode} is not supported`);}}#f(Vt){let e1=Vt.indexOf(" ");if(e1===-1)return ft.fromIDCode(Vt);let t1=Vt.slice(0,e1),g1=Vt.slice(e1+1);return ft.fromIDCode(t1,g1);}#r(){let Vt=this.#f(this.idcode);Vt.setFragment(this.fragment),this.#e.setMolecule(Vt);}#n(){let Vt=dt.decode(this.idcode,{ensureCoordinates:true})??Bt.create();Vt.setFragment(this.fragment),this.#e.setReaction(Vt);}#i=false;#o(Vt){this.#i=true;try{Vt();}finally{this.#i=false;}}#l=Vt=>{let e1=this.idcode,t1=this.fragment;this.#o(()=>{if(Vt.type==="molecule")switch(this.mode){case It.MODE.MOLECULE:{let c1=this.getMolecule();this.idcode=`${c1.getIDCode()} ${c1.getIDCoordinates()}`,this.fragment=c1.isFragment();break;}case It.MODE.REACTION:{let c1=this.getReaction();this.idcode=dt.encode(c1,{keepAbsoluteCoordinates:true,mode:dt.INCLUDE_MAPPING|dt.INCLUDE_COORDS|dt.RETAIN_REACTANT_AND_PRODUCT_ORDER}),this.fragment=c1.isFragment();break;}default:throw new Error(`Unsupported mode ${this.mode}`);}});let g1=new CustomEvent("change",{detail:Vt,bubbles:true});if(this.dispatchEvent(g1),Vt.mode==="molecule"){if(this.idcode!==e1){let c1=new CustomEvent("idcode-changed",{detail:this.idcode,bubbles:true});this.dispatchEvent(c1);}if(this.fragment!==t1){let c1=new CustomEvent("fragment-changed",{detail:this.fragment,bubbles:true});this.dispatchEvent(c1);}}};#D(){this.#e&&(this.#e.destroy(),this.#e=void 0);}#u(){this.#D(),this.#P();}connectedCallback(){this.#t={idcode:this.getAttribute("idcode")||"",fragment:this.hasAttribute("fragment"),mode:this.getAttribute("mode")||It.MODE.MOLECULE,readonly:this.hasAttribute("readonly")},this.#P();}disconnectedCallback(){this.#D();}adoptedCallback(){this.connectedCallback();}attributeChangedCallback(Vt,e1,t1){if(!this.#e||this.#i)return;(()=>{switch(Vt){case "idcode":return this.#t.idcode=String(t1),()=>this.#A();case "fragment":return this.#t.fragment=t1!==null,()=>this.#A();case "mode":return this.#t.mode=String(t1),()=>this.#u();case "readonly":return this.#t.readonly=t1!==null,()=>this.#u();default:throw new Error("unsupported attribute change");}})()();}}return It;}function N8(j){let{GenericEditorArea:ft,GenericEditorToolbar:Bt,GenericUIHelper:dt,Molecule:It,Reaction:v1,ReactionEncoder:Vt}=j,e1=ju(ft,Bt,dt,It,v1);function t1(){let g1=customElements.get("openchemlib-editor");if(g1)return g1;let c1=Bu(e1,It,v1,Vt);customElements.define("openchemlib-editor",c1);let p1=document.createElement("style");return p1.id="openchemlib-editor-default-style",p1.innerHTML=`
    /* dynamicaly added from openchemlib registerCustomElement with low priority */
    openchemlib-editor:defined {
      display: block;
      height: 400px;
      width: 600px;
    }
    `,document.head.prepend(p1),c1;}j.CanvasEditor=e1,j.registerCustomElement=t1,delete j.GenericEditorArea,delete j.GenericEditorToolbar,delete j.GenericUIHelper;}function mu(j){j.prototype.getNextCustomAtomLabel=function(Bt){let dt=Bt||"1",It=new Set();for(let Vt=0;Vt<this.getAllAtoms();Vt++){let e1=this.getAtomCustomLabel(Vt);e1&&It.add(e1);}let v1=0;for(;It.has(dt)&&v1++<100;)dt=o2(dt);return dt;};}function o2(j){let ft=j.match(/(\d+)/);if(ft){let dt=Number.parseInt(ft[1],10);return j.replace(/(\d+)/,(dt+1).toString());}let Bt=j.match(/([a-yA-Y])([^a-zA-Z]*)$/);if(Bt){let dt=Bt[1],It=String.fromCodePoint(dt.codePointAt(0)+1);return It==="Z"||It==="z"?"1":j.replace(/([a-yA-Y])([^a-zA-Z]*)$/,`${It}$2`);}return "1";}function JP(j,ft){switch(ft){case "superscript":for(let Bt=0;Bt<j.getAllAtoms();Bt++){let dt=j.getAtomCustomLabel(Bt);dt&&!dt.startsWith("]")&&j.setAtomCustomLabel(Bt,`]${dt}`);}break;case "normal":for(let Bt=0;Bt<j.getAllAtoms();Bt++){let dt=j.getAtomCustomLabel(Bt);dt?.startsWith("]")&&j.setAtomCustomLabel(Bt,dt.slice(1));}break;case "auto":for(let Bt=0;Bt<j.getAllAtoms();Bt++){let dt=j.getAtomCustomLabel(Bt);dt&&(j.getAtomLabel(Bt)==="C"?dt.startsWith("]")&&j.setAtomCustomLabel(Bt,dt.slice(1)):dt.startsWith("]")||j.setAtomCustomLabel(Bt,`]${dt}`));}break;}}function ku(j){let ft=j.fromMolfile;j.fromMolfile=function(dt,It={}){let{customLabelPosition:v1}=It,Vt=ft.call(this,dt),e1=dt.includes(`\r
`)?`\r
`:`
`,t1=dt.split(e1);if(t1.length<4||!t1[3].includes("V2000"))return Vt;let g1=t1.slice(4+Vt.getAllAtoms()+Vt.getAllBonds());for(let c1=0;c1<g1.length;c1++){let p1=g1[c1];if(p1.startsWith("A  ")){let l1=Number(p1.slice(3)),U0=g1[c1+1]?.trim();c1++,!Number.isNaN(l1)&&l1<=Vt.getAllAtoms()&&U0&&!Vt.getAtomCustomLabel(l1-1)&&Vt.setAtomCustomLabel(l1-1,U0);}if(p1.startsWith("V  ")){let l1=p1.split(" ").filter(Boolean);if(l1.length>=3){let U0=Number(l1[1]),u0=l1.slice(2).join(" ");!Number.isNaN(U0)&&U0<=Vt.getAllAtoms()&&!Vt.getAtomCustomLabel(U0-1)&&Vt.setAtomCustomLabel(U0-1,u0);}}}return JP(Vt,v1),Vt;};}var l2=["M  STY","M  SLB","M  SAL","M  SDT","M  SDD","M  SED"];function Fu(j){let ft=j.prototype.toMolfile;j.prototype.toMolfile=function(dt={}){let It=this.getCompactCopy(),{includeCustomAtomLabelsAsALines:v1=false,includeCustomAtomLabelsAsVLines:Vt=false,customLabelPosition:e1,removeCustomAtomLabels:t1=false}=dt;JP(It,e1);let g1=ft.call(It);if(!v1&&!Vt&&!t1)return g1;let c1=g1.includes(`\r
`)?`\r
`:`
`,p1=g1.split(c1);if(t1&&(p1=p1.filter(u0=>!l2.some($1=>u0.startsWith($1)))),p1.length<4||!p1[3].includes("V2000"))return g1;let l1=[];for(let u0=0;u0<It.getAllAtoms();u0++){let $1=It.getAtomCustomLabel(u0);if($1){let M0=String(u0+1).padStart(3," ");v1&&l1.push(`A  ${M0}`,$1),Vt&&l1.push(`V  ${M0} ${$1}`);}}let U0=p1.findIndex(u0=>u0.startsWith("M  END"));return U0===-1?g1:(p1.splice(U0,0,...l1),p1.join(c1));};}function vu(j){let{ConformerGenerator:ft,ForceFieldMMFF94:Bt,Molecule:dt}=j;ft.prototype.molecules=function*(){let t1;for(;(t1=this.getNextConformerAsMolecule())!==null;)yield t1;};let It={maxIts:4e3,gradTol:1e-4,funcTol:1e-6},v1=Bt.prototype._minimise;delete Bt.prototype._minimise,Bt.prototype.minimise=function(t1){return t1={...It,...t1},v1.call(this,t1.maxIts,t1.gradTol,t1.funcTol);},ku(dt),Fu(dt),mu(dt);function Vt(e1){if(!e1)return null;if(e1.includes("V2000")||e1.includes("V3000"))return dt.fromMolfile(e1);try{return dt.fromSmiles(e1);}catch{}try{return dt.fromIDCode(e1);}catch{}return null;}dt.fromText=function(t1){let g1=Vt(t1);return g1&&g1.getAllAtoms()>0?g1:null;},dt.prototype.getOCL=function(){return j;};}function Fn(){}function Q2(j){let ft=j.document,Bt={},dt={userAgent:"webkit"},It=Fn;It.__moduleStartupDone=Fn;var t1="object",g1="anonymous",c1="fnStack",p1=`
`,l1={4:1,9:1,5:1,8:1},U0="Unknown",u0="boolean",$1="number",M0="string",St=2147483647,He='For input string: "',re="null",VA="__noinit__",C1={4:1,5:1,8:1},B0={4:1,19:1,33:1,27:1},vn=" (copy)",st={4:1,9:1,14:1,5:1,11:1,8:1,13:1},q1=65536,r1=65535,Y0=1e4,tA="fromIndex: ",I8=" > toIndex: ",ZP=", toIndex: ",x8=", length: ",_P="Index: ",WP=", Size: ",jt={15:1,4:1,5:1},fn={4:1},H={7:1,4:1,5:1},We=16777215,J0=65280,Dn=.30000001192092896,Me=545460846592,Qe=4294967296,Se=8589934592,qe=17179869184,Ve=34359738368,Ee=68719476736,Ne=137438953472,un=274877906944,ht={12:1,4:1,5:1},r0=536870912,$P=2.617993878,aP=3.665191429,X1=6.283185307179586,n1=3.141592653589793,m0=1.5707963267948966,x0=16384,EA=-8355712,Y1=4096,eA={l:0,m:0,h:1},nA={l:0,m:0,h:2},AA={l:0,m:0,h:4},Z0=1920,i0=1024,je=234881024,PA=167772160,on=100663296,ln=201326592,$e=114688,ae=4063232,rA=3801088,pn=3538944,Rn=3014656,iA=3145728,F0=2097152,tn=393216,ie=917504,Qn=1966080,Ln=0xf8000000000,NA=0xf0000000000,IA=0xe8000000000,xA=0xd8000000000,OA=0xb8000000000,tr=0xe0000000000,er=0xc0000000000,fA=8796093022208,Cn=549755813888,XA=1649267441664,GA=3848290697216,DA=8246337208320,nr=0xc8000000000,Ar=9345848836096,Pr=0x98000000000,Be=29360128,Ie=268435456,Q0=8192,z1=2048,uA=-1.5707963267948966,Le=130560,oA=6291456,rr=-2.0943951023931953,b0=1.0471975511965976,Hn=.5235987755982988,U1=786432,ye=524288,Ue=262144,O0=-16777216,fe=-65536,KA=-32768,ir='<DrawingObject type="',YA="></DrawingObject>",Kt={4:1,5:1,16:1},n0={4:1,5:1},fr="unsupported atomicNo:",O8=2.8415926535897933,S1={28:1,4:1,9:1,5:1,8:1},Mn={4:1,5:1,11:1,8:1},J={10:1,4:1,5:1},c0={4:1,9:1,5:1,8:1,30:1},A0=131072,JA=196608,ZA={l:6,m:0,h:4},se=1048576,he=3221225472,Dr={l:0,m:0,h:3},yn=8388608,ur={l:0,m:0,h:120},_A={36:1,43:1,38:1},s0={4:1,9:1,5:1,11:1,8:1},X8={36:1,43:1,95:1,38:1,83:1},WA={4:1,5:1,36:1,43:1,95:1,20:1,38:1,83:1,127:1},L0=17976931348623157e292,or=-17976931348623157e292,xe={87:1,74:1,75:1,4:1,9:1,5:1,8:1},R1="??",Te={58:1,4:1,9:1,5:1,8:1},be=2013265920,k0=32505856,S0=31744,$A=122880,Sn={129:1,4:1,9:1,5:1,11:1,8:1},lA=33554432,q0=67108864,Oe=32768,aA=-131073,tP=-33554433,eP=-33292289,lr=-3.141592653589793,W1=4194303,V0=3072,w0={46:1,4:1,5:1},Qr=.7853981633974483,G8=3.061592653589793,E0=1073741824,K8=541165879296,qt={17:1,4:1,5:1,8:1},nP=281.17,AP=289.19,PP=294.21,rP="0.0000",iP="NOSEARCH_OCL_CUSTOM_LABEL",me="M  END",qn="ATOMS",P1=-2147483648,QA="$$$$",LA="M  V30 ",Y8="M  V30 COUNTS ",J8="M  V30 MDLV30/STEREL",Z8="M  V30 MDLV30/STERAC",Lr="M  V30 END CTAB",_8=34028234663852886e22,yr={l:2,m:0,h:4},_0=4194304,Ur={l:4179966,m:4063288,h:7},W8={l:4194302,m:4193343,h:127},J1='" ',sr='stroke="',fP='stroke-width="',DP='" />',$8='class="event" ',a8='opacity="0" />',W0=16711680,t9="' position:",e9="'. Position:",n9=1099511627776,A9=2199023255552,P9=4398046511104,hr="', position:",r9="Assignment of aromatic double bonds failed",yA=2147483646,i9={102:1,4:1},Tr="Members of ESR groups must only be stereo centers with known configuration.",br="Over- or under-specified stereo feature or more than one racemic type bond",cr="Ambiguous configuration at stereo center because of 2 parallel bonds",f9={114:1,88:1,25:1,4:1},uP=1e5,UA=1.399999976158142,Vn=3.1415927410125732,wr=1.5707963705062866,D9=.029999999329447746,u9=6.2831854820251465,o9=1.9106119871139526,l9=2.2639999389648438,gr=2.16759991645813,Q9=2.0662999153137207,L9=2.3561999797821045,y9=2.2844998836517334,U9=1.9106120066311025,s9=2.094395160675049,sA=.6000000238418579,oP=1.7999999523162842,zr=.05000000074505806,P0={4:1,9:1,5:1,8:1,37:1},lP=.2617993950843811,En=2.0943951023931953,hA=1.2000000476837158,dr=1.4299999475479126,QP=1.659999966621399,jr=1.4600000381469727,Br=1.899999976158142,h9=1.8200000524520874,T9=2.4600000381469727,b9=2.4200000762939453,LP=2.450000047683716,mr=2.440000057220459,c9=2.4000000953674316,kr=2.319999933242798,Fr=2.2899999618530273,w9=1.8600000143051147,g9=2.5299999713897705,z9=2.490000009536743,d9=2.4700000286102295,j9=2.880000114440918,B9=2.9200000762939453,m9=2.869999885559082,vr=2.8299999237060547,k9=2.7899999618530273,F9=2.809999942779541,v9=2.799999952316284,p9=3.049999952316284,pr=.3199999928474426,R9=1.3300000429153442,TA=.8500000238418579,C9=1.600000023841858,H9=1.3899999856948853,Rr=1.159999966621399,M9=1.1100000143051147,S9=1.0700000524520874,q9=1.9600000381469727,V9=1.7100000381469727,Cr=1.3600000143051147,E9=1.340000033378601,N9=1.2200000286102295,I9=1.2400000095367432,yP=1.2100000381469727,x9=2.0999999046325684,Hr=1.850000023841858,Mr=1.6299999952316284,O9=1.4700000286102295,X9=1.2799999713897705,G9=1.440000057220459,K9=1.7599999904632568,Y9=1.7300000190734863,J9=1.7200000286102295,UP=1.6799999475479126,Z9=1.690000057220459,_9=1.6699999570846558,Sr=1.649999976158142,bA=1.7000000476837158,W9=1.6200000047683716,$9=1.3700000047683716,a9=1.2899999618530273,ti=1.4500000476837158,ei=1.5700000524520874,qr=-0.5235987755982988,cA=-1.0471975511965976,ni=-0.7853981633974483,Ai=.017453292519943295,Pi=.17453292519943295,wA=1e-4,Un={148:1},sn=57.29577951308232,ri=.021922173386725217,gA=1e-8,De=2.5120960256267386,zA=1e-5,sP="MMFF94s+",ii="nonbonded cutoff",fi="dielectric constant",Di="dielectric model",ui="angle bend",oi="bond stretch",li="electrostatic",Qi="out of plane",Li="stretch bend",yi="torsion angle",Ui="van der waals",si=.07000000000000006,hi=.1200000000000001,Ti=-7.840000000000001,hP="OCL_RXN_V1.0:",bi="$RXN V3000",ci="'$MOL' tag not found",wi="'M  END' not found",gi="M  V30 BEGIN REACTANT",zi=`
Actelion Java MolfileCreator 2.0

  0  0  0  0  0  0              0 V3000
`,di="M  V30 BEGIN PRODUCT",ji=2475109711874,Bi=2475109711875,mi=2887426572290,ki=2887426572291,Fi=2887627898882,vi=2887627898883,pi=2887627997186,Ri=3437182386178,Ci=3437182386179,Hi=3437383712770,Mi=3437383811074,Si=3437652148226,qi=3437652377602,Vi=4674132967426,Ei=4674334294018,Ni=4674334392322,Ii=4674602729474,xi=4674602958850,Oi=5086449827842,Xi=5086449827848,Gi=5086651154434,Ki=5086651252738,Yi=5086651252743,Ji=5086919589890,Zi=5086919688194,_i=5636406968322,Wi=5636407066626,$i=5636675502082,ai={l:589826,m:590112,h:4},tf={l:589827,m:590112,h:4},ef={l:589826,m:590160,h:4},nf={l:589827,m:590160,h:4},Af={l:688130,m:590160,h:4},Pf={l:589826,m:590224,h:4},rf={l:589827,m:590224,h:4},ff={l:819203,m:590224,h:4},Df={l:589826,m:590368,h:4},uf={l:688130,m:590368,h:4},of={l:819202,m:590368,h:4},lf={l:819208,m:590368,h:4},Qf={l:589826,m:590416,h:4},Lf={l:589832,m:590416,h:4},yf={l:688130,m:590416,h:4},Uf={l:819202,m:590416,h:4},sf={l:589826,m:598304,h:4},hf={l:589827,m:598304,h:4},Tf={l:688130,m:598304,h:4},bf={l:1114114,m:598304,h:4},cf={l:1212418,m:598304,h:4},wf={l:589826,m:598308,h:4},gf={l:589826,m:688416,h:4},zf={l:589826,m:688464,h:4},df={l:589826,m:688720,h:4},jf={l:688130,m:688720,h:4},Bf={l:589826,m:590112,h:288},mf={l:589826,m:590112,h:336},kf={l:589826,m:688416,h:336},Ff={l:589826,m:688464,h:336},vf={l:688130,m:688464,h:336},pf={l:589826,m:590112,h:400},Rf={l:589826,m:688416,h:400},Cf={l:589826,m:688464,h:400},Hf={l:688130,m:688464,h:400},Mf={l:589826,m:819488,h:400},Sf={l:589826,m:819536,h:400},qf={l:589826,m:819600,h:400},Vf={l:688130,m:819600,h:400},Ef={l:819202,m:819600,h:400},Nf={l:589831,m:688416,h:592},If={l:589831,m:688464,h:592},xf={l:688135,m:688464,h:592},Of={l:589831,m:819536,h:592},Xf={l:688135,m:819536,h:592},Gf={l:688135,m:819600,h:592},Kf={l:589832,m:1212704,h:592},Yf={l:589832,m:1212752,h:592},Jf={l:688136,m:1212752,h:592},Zf={l:589831,m:688464,h:656},_f={l:589826,m:590112,h:8480},Wf={l:589826,m:688416,h:8480},$f={l:589826,m:688464,h:8480},af={l:688130,m:688464,h:8480},tD={l:819202,m:819600,h:8480},eD={l:688135,m:1212752,h:8480},nD={l:589832,m:1213008,h:8480},AD={l:688136,m:1213008,h:8480},PD={l:589826,m:590112,h:8484},rD={l:589826,m:590160,h:8484},iD={l:688130,m:598304,h:8484},fD=.6262000203132629,DD=-1.3825000524520874,TP=.33169999718666077,uD=-1.4915000200271606,oD=.3540000021457672,lD=.38179999589920044,QD=-0.6019999980926514,LD=.07999999821186066,yD=-0.7379999756813049,UD="Atom-types are 64-bit numbers describing atoms and their near surrounding.",Vr="Recognized atom types and their contributions are:",sD="Druglikeness predictor not properly initialized.",hD=3.009999990463257,TD=-0.17000000178813934,bD=-0.1809999942779541,cD=-0.2029999941587448,dA=2147483648,wD="ene-Metathesis",gD="Halogenation ring closure",zD=" (rootPairSets:",Er=-1e10,bP={204:1,4:1,9:1,5:1,8:1,30:1},cP=-99999,ce="any",dD="is not in a ring",Nr="Excluded atoms:",wP="exactly 1",gP="exactly 2",Ir="less than 3",jD="less than 4",zP="at least 1",xr="at least 2",Or="at least 3",BD="at least 4",Xr="1 or 2",mD="less than 2",kD="Allowed atoms:",hn={137:1},FD="Atom Properties",dP=-8346432,vD="/html/editor/editor.html",pD="Structure Editor Help",jP="Dummy look and feel",Nn="Substance Graphite",In="Radiance Graphite",xn="Radiance Night",RD=.6499999761581421,CD=.3499999940395355,Gr="editorButtons.png",HD="esrButtons.png",Tn=.800000011920929,Kr=-1774190527,Yr=-1438646166,On=-1433403286,BP=-1431655766,mP=-1448432982,ke=1073741866,bn=1073741930,Jr="MMFF94",Zr="MMFF94s",MD="/resources/forcefield/mmff94/94s/outofplane.csv",Xn="undefined",_r={171:1,4:1,5:1,8:1},SD=.08726646502812703,qD="overlapping",Wr=.003921568859368563,p0=1048575,$0=17592186044416,cn=1e9,Gn=-17592186044416,g0=5.56,f0=11.12,Fe=13.34,wn=14.44,jA=.3010299956639812,VD="BigInteger divide by zero",D1=4294967295,ED={l:0,m:0,h:524288},kP={32:1,115:1},$r={107:1,110:1},Kn={36:1,43:1,38:1,85:1},FP=16777619,ND={36:1,38:1,85:1},vP="delete",pP=15525485,ID=5960464477539063e-23,ar=16777216,RP={4:1,32:1,56:1,80:1},t8="Invalid UTF8 sequence",e8=.20000000298023224,xD=" distance:",OD={4:1,5:1,11:1,8:1,452:1},XD="gGP`@dfyjidNcGI[WQCP`<",Yn={48:1},l={},T={},T1={},Xt={},w={},ue={},n={},U={},E={},Yt={},Z={},we={},Y={},At={},i1={},A1={},nt={},tt={},a={},q={},Tt={},j1={},B={},W={},R={},N1={},lt={},F1={},z0={},Rt={},p={},V1={},o0={},l0={},X0={},gt={},_t={},_={},$={},y,BA,CP;l.Beb=function(){};function Cu(f,t){typeof window===t1&&typeof window.$gwt===t1&&(window.$gwt[f]=t);}function Hu(f,t,e,A){l.ofb();var P=CP;function r(){for(var i=0;i<P.length;i++)P[i]();}if(f)try{eu(r)();}catch(i){f(t,i);}else eu(r)();}l.ofb=function(){CP==null&&(CP=[]);};function Mu(){l.ofb();for(var f=CP,t=0;t<arguments.length;t++)f.push(arguments[t]);}l.mfb=function(){};function Jn(){}function Zn(f){return Array.isArray(f)&&f.wd===Jn?T.Uc(f):f.toString();}function E1(f,t){var e=j;if(f==="")return e;var A=f.split(".");if(!(A[0]in e)&&e.execScript&&e.execScript("var "+A[0]),t){var P=t.prototype.ud;P.e=t;}for(var r;A.length&&(r=A.shift());)e=e[r]=e[r]||!A.length&&t||{};return e;}l.zfb=function(t){function e(){}return e.prototype=t||{},new e();},l.yfb=function(t){return t instanceof Array?t[0]:null;};function h0(f,t,e){var A=function(){return f.apply(A,arguments);};return t.apply(A,e),A;}function o1(){}function a0(f,t){for(var e in t)t[e].configurable=true;Object.defineProperties(f,t);}function x(f,t,e){var A=BA,P=A[f],r=l.yfb(P);P&&!r?y=P:(y=l.tfb(t),y.vd=e,!t&&(y.wd=Jn),A[f]=y);for(var i=3;i<arguments.length;++i)arguments[i].prototype=y;r&&(y.ud=r);}l.tfb=function(t){var e=t&&t.prototype;return !e&&(e=BA[t]),l.zfb(e);},l.sfb=function(){},j.goog=j.goog||{},j.goog.global=j.goog.global||j,BA={},T.Jc=function(t,e){return k1(t)===k1(e);},T.Kc=function(t){return t.ud;},T.Lc=function(t){return w.Dzb(t);};function GD(){}T.Oc=function(t,e){return ve(t)?T.ejb(t,e):gn(t)?T.Whb(t,e):mA(t)?T.Ogb(t,e):A8(t)?t.lb(e):n8(t)?T.Jc(t,e):T1.y4(t,e);},T.Qc=function(t){return ve(t)?T.ijb():gn(t)?T.Xhb():mA(t)?T.Pgb():A8(t)?t.ud:n8(t)?T.Kc(t):T1.z4(t);},T.Sc=function(t){return ve(t)?T.jjb(t):gn(t)?T.Yhb(t):mA(t)?T.Qgb(t):A8(t)?t.nb():n8(t)?T.Lc(t):T1.A4(t);},T.Uc=function(t){var e;return T.qhb(T.Qc(t))+"@"+(e=w.Xzb(w.Wzb(T.Sc(t))),e.toString(16));},x(1,null,{},GD),y.lb=function(t){return T.Jc(this,t);},y.mb=function(){return T.Kc(this);},y.nb=function(){return T.Lc(this);},y.ob=function(){var t;return T.qhb(T.Qc(this))+"@"+(t=w.Xzb(w.Wzb(T.Sc(this))),t.toString(16));},y.equals=function(f){return this.lb(f);},y.hashCode=function(){return this.nb();},y.toString=function(){return this.ob();},T1.y4=function(t,e){return T1.E4(t)?T1.B4(t,e):k1(t)===k1(e);},T1.z4=function(t){return t.ud||Array.isArray(t)&&k(l.Vab,1)||l.Vab;},T1.A4=function(t){return T1.F4(t)?T1.C4(t):w.Dzb(t);},T1.B4=function(t,e){return t.equals(e);},T1.C4=function(t){return t.hashCode();},T1.D4=function(){return [];},T1.E4=function(t){return !!t&&!!t.equals;},T1.F4=function(t){return !!t&&!!t.hashCode;},T1.G4=function(t){return t.toString?t.toString():"[JavaScriptObject]";},Xt.k5=function(){Xt.k5=o1;var t,e;e=!Xt.t5(),t=new Xt.C5(),Xt.j5=e?new Xt.u5():t;},Xt.l5=function(t){Xt.k5(),Xt.j5.Lc(t);},Xt.m5=function(t){Xt.k5();var e;return e=Xt.j5.Mc(t),Xt.n5(e);},Xt.n5=function(t){var e,A,P,r;for(e="l5",A="YB",r=j.Math.min(t.length,5),P=r-1;P>=0;P--)if(T.ejb(t[P].d,e)||T.ejb(t[P].d,A)){t.length>=P+1&&w.Xzb(t).splice(0,P+1);break;}return t;},Xt.o5=function(t){var e=/function(?:\s+([\w$]+))?\s*\(/,A=e.exec(t);return A&&A[1]||g1;},Xt.p5=function(t){return Xt.k5(),t&&t[c1]?t[c1]:[];},Xt.q5=function(t){return Xt.k5(),t.name||(t.name=Xt.o5(t.toString()));},Xt.r5=function(t){return Xt.k5(),parseInt(t)||-1;},Xt.s5=function(t){Xt.k5();var e=t.backingJsObject;if(e&&e.stack){var A=e.stack,P=e+p1;return A.substring(0,P.length)==P&&(A=A.substring(P.length)),A.split(p1);}return [];},Xt.t5=function(){return Error.stackTraceLimit>0?(j.Error.stackTraceLimit=Error.stackTraceLimit=64,true):"stack"in new Error();},x(436,1,{}),Xt.u5=function(){},x(305,436,{},Xt.u5),y.Lc=function(t){var e={},A=[];t[c1]=A;for(var P=arguments.callee.caller;P;){var r=Xt.q5(P);A.push(r);var i=":"+r,D=e[i];if(D){var u,o;for(u=0,o=D.length;u<o;u++)if(D[u]===P)return;}(D||(e[i]=[])).push(P),P=P.caller;}},y.Mc=function(t){var e,A,P,r;for(P=Xt.p5(t),A=T1.M4(P),r=z(l.Obb,l1,90,A,0,1),e=0;e<A;e++)r[e]=new T.Uib(T1.L4(P,e),null,-1);return r;},Xt.x5=function(t,e){var A,P,r,i,D,u,o,Q,L;return Q="",w.Xzb(e).length==0?t.Nc(U0,g1,-1,-1):(L=T.wjb(e),T.ejb(w.Xzb(L).substr(0,3),"at ")&&(L=(w.Szb(3,w.Xzb(L).length+1),w.Xzb(L).substr(3))),L=Xt.y5(L),D=w.Xzb(L).indexOf("("),D==-1?(D=w.Xzb(L).indexOf("@"),D==-1?(Q=L,L=""):(Q=T.wjb((w.Szb(D+1,w.Xzb(L).length+1),w.Xzb(L).substr(D+1))),L=T.wjb((w.Rzb(0,D,w.Xzb(L).length),w.Xzb(L).substr(0,D))))):(A=w.Xzb(L).indexOf(")",D),Q=(w.Rzb(D+1,A,w.Xzb(L).length),w.Xzb(L).substr(D+1,A-(D+1))),L=T.wjb((w.Rzb(0,D,w.Xzb(L).length),w.Xzb(L).substr(0,D)))),D=T.ljb(L,Z1(46)),D!=-1&&(L=(w.Szb(D+1,w.Xzb(L).length+1),w.Xzb(L).substr(D+1))),(w.Xzb(L).length==0||T.ejb(L,"Anonymous function"))&&(L=g1),u=T.ojb(Q,Z1(58)),r=T.pjb(Q,Z1(58),u-1),o=-1,P=-1,i=U0,u!=-1&&r!=-1&&(i=(w.Rzb(0,r,w.Xzb(Q).length),w.Xzb(Q).substr(0,r)),o=Xt.r5((w.Rzb(r+1,u,w.Xzb(Q).length),w.Xzb(Q).substr(r+1,u-(r+1)))),P=Xt.r5((w.Szb(u+1,w.Xzb(Q).length+1),w.Xzb(Q).substr(u+1)))),t.Nc(i,L,o,P));},Xt.y5=function(t){return t.replace(/\[.*?\]/g,"");},x(437,436,{}),y.Lc=function(t){},y.Nc=function(t,e,A,P){return new T.Uib(e,t+"@"+P,A<0?-1:A);},y.Mc=function(t){var e,A,P,r,i,D;if(r=Xt.s5(t),i=z(l.Obb,l1,90,0,0,1),e=0,P=T1.M4(r),P==0)return i;for(D=Xt.x5(this,T1.L4(r,0)),T.ejb(D.d,g1)||(i[e++]=D),A=1;A<P;A++)i[e++]=Xt.x5(this,T1.L4(r,A));return i;},Xt.C5=function(){},x(306,437,{},Xt.C5),y.Nc=function(t,e,A,P){return new T.Uib(e,t,-1);},l.E5=function(t){return t;};function k(f,t){return l.G5(f,t);}l.G5=function(t,e){return T.Bhb(t,e);},l.H5=function(t){return t.__elementTypeCategory$==null?10:t.__elementTypeCategory$;},l.I5=function(t){return t.__elementTypeId$;};function h1(f,t,e,A,P,r){return l.K5(f,t,e,A,P,0,r);}l.K5=function(t,e,A,P,r,i,D){var u,o,Q,L,s;if(L=r[i],Q=i==D-1,u=Q?P:0,s=l.M5(u,L),P!=10&&F(k(t,D-i),e[i],A[i],u,s),!Q)for(++i,o=0;o<L;++o)l.O5(s,o,l.K5(t,e,A,P,r,i,D));return s;};function z(f,t,e,A,P,r){var i;return i=l.M5(P,A),P!=10&&F(k(f,r),t,e,P,i),i;}l.M5=function(t,e){var A=new Array(e),P;switch(t){case 14:case 15:P=0;break;case 16:P=false;break;default:return A;}for(var r=0;r<e;++r)A[r]=P;return A;};function n8(f){return Array.isArray(f)&&f.wd===Jn;}l.O5=function(t,e,A){return t[e]=A;};function a1(f,t,e){return l.O5(f,t,e);}l.Q5=function(t,e){t.ud=e;},l.R5=function(t,e){t.__elementTypeCategory$=e;},l.S5=function(t,e){t.__elementTypeId$=e;};function F(f,t,e,A,P){return l.Q5(P,f),P.vd=t,P.wd=Jn,l.S5(P,e),l.R5(P,A),P;}l.U5=function(t,e){return l.H5(e)!=10&&F(T.Qc(e),e.vd,l.I5(e),l.H5(e),t),l.E5(t);};function Su(f,t){return ve(f)?!!_D[t]:f.vd?!!f.vd[t]:gn(f)?!!ZD[t]:mA(f)?!!JD[t]:false;}function A8(f){return !Array.isArray(f)&&f.wd===Jn;}function Jt(f,t){return f!=null&&Su(f,t);}function mA(f){return typeof f===u0;}function gn(f){return typeof f===$1;}function KD(f){return f!=null&&l.X6(f)&&f.wd!==Jn;}function YD(f,t){return l.Y6(f,t);}function ve(f){return typeof f===M0;}l.X6=function(t){return typeof t===t1||typeof t=="function";},l.Y6=function(t,e){return t&&e&&t instanceof e;};function k1(f){return f??null;}function Ft(f){return Math.max(Math.min(f,St),-2147483648)|0;}var JD,ZD,_D;l.Ceb=function(t){return t&&t.__java$exception;};function H1(f){var t;return Jt(f,27)?f:(t=l.Ceb(f),t||(t=new T1.s4(f),Xt.l5(t)),t);}function Ut(f){return f.backingJsObject;}function qu(f){var t;return f==null?false:(t=typeof f,T.ejb(t,u0)||T.ejb(t,$1)||T.ejb(t,M0)||f.$implements__java_io_Serializable||Array.isArray(f));}function Vu(f){var t;return t=typeof f,T.ejb(t,u0)||T.ejb(t,$1)||T.ejb(t,M0)?true:f!=null&&f.$implements__java_lang_Comparable;}T.Kgb=function(){T.Kgb=o1,kA=w.Xzb(false),w.Xzb(true);},T.Lgb=function(t){return w.Yzb((w.Lzb(t),t));},T.Mgb=function(t,e){return Nu(w.Yzb((w.Lzb(t),t)),w.Yzb((w.Lzb(e),e)));},T.Ngb=function(t,e){return T.Mgb(t,e);},T.Ogb=function(t,e){return w.Lzb(t),k1(t)===k1(e);},T.Pgb=function(){return l.wbb;},T.Qgb=function(t){return w.Yzb((w.Lzb(t),t))?1231:1237;};function Eu(f){return T.Kgb(),T.ejb(u0,typeof f);}function Nu(f,t){return T.Kgb(),f==t?0:f?1:-1;}T.Tgb=function(t,e){return T.Kgb(),ve(t)?T.Zib(t,e):gn(t)?T.Uhb(t,e):mA(t)?T.Ngb(t,e):t.Rb(e);},JD={4:1,300:1,32:1};var kA;function Iu(f){return T.ejb(typeof f,M0)?true:f!=null&&f.$implements__java_lang_CharSequence;}T.ohb=function(t){t.i=Xu++;},T.phb=function(t){t.n==null&&T.Fhb(t);},T.qhb=function(t){return T.phb(t),t.n;},T.rhb=function(t){return (t.f&4)!=0;},T.shb=function(t){return (t.f&1)!=0;},T.thb=function(){T.ohb(this),this.n=null,this.j=null,this.g=null,this.d=null,this.b=null,this.k=null,this.a=null;},T.vhb=function(t){var e;return e=new T.thb(),e.n="Class$"+(t?"S"+t:""+e.i),e.b=e.n,e.j=e.n,e;};function X(f){var t;return t=T.vhb(f),T.Jhb(f,t),t;}function Xe(f,t){var e;return e=T.vhb(f),T.Jhb(f,e),e.f=t?8:0,e;}function xu(){var f;return f=T.vhb(null),f.f=2,f;}function en(f){var t;return t=T.vhb(f),t.k=f,t.f=1,t;}T.Bhb=function(t,e){var A=t.a=t.a||[];return A[e]||(A[e]=t.Tc(e));};function Ou(f){if(f.Yc())return null;var t=f.k;return BA[t];}T.Fhb=function(t){if(t.Xc()){var e=t.c;e.Yc()?t.n="["+e.k:e.Xc()?t.n="["+e.Vc():t.n="[L"+e.Vc()+";",t.b=e.Uc()+"[]",t.j=e.Wc()+"[]";return;}var A=t.g,P=t.d;P=P.split("/"),t.n=T.Ihb(".",[A,T.Ihb("$",P)]),t.b=T.Ihb(".",[A,T.Ihb(".",P)]),t.j=P[P.length-1];},T.Ihb=function(t,e){for(var A=0;!e[A]||e[A]=="";)A++;for(var P=e[A++];A<e.length;A++)!e[A]||e[A]==""||(P+=t+e[A]);return P;},T.Jhb=function(t,e){if(t){e.k=t;var P=Ou(e);if(!P){BA[t]=[e];return;}P.ud=e;}},x(207,1,{},T.thb),y.Tc=function(t){var e;return e=new T.thb(),e.f=4,t>1?e.c=T.Bhb(this,t-1):e.c=this,e;},y.Uc=function(){return T.phb(this),this.b;},y.Vc=function(){return T.qhb(this);},y.Wc=function(){return T.phb(this),this.j;},y.Xc=function(){return T.rhb(this);},y.Yc=function(){return T.shb(this);},y.ob=function(){return ((this.f&2)!=0?"interface ":(this.f&1)!=0?"":"class ")+(T.phb(this),this.n);},y.f=0,y.i=0;var Xu=1;T.Ohb=function(t){return T.ejb($1,typeof t)||YD(t,j.java.lang.Number$impl);},T.Phb=function(t){if(T.Nhb==null&&(T.Nhb=new RegExp("^\\s*[+-]?(NaN|Infinity|((\\d+\\.?\\d*)|(\\.\\d+))([eE][+-]?\\d+)?[dDfF]?)\\s*$")),!T.Nhb.test(t))throw Ut(new T.Tib(He+t+'"'));return parseFloat(t);},T.Qhb=function(t,e,A){var P,r,i,D,u;if(t==null)throw Ut(new T.Tib(re));for(i=w.Xzb(t).length,D=i>0&&(w.Szb(0,w.Xzb(t).length),w.Xzb(t).charCodeAt(0)==45||(w.Szb(0,w.Xzb(t).length),w.Xzb(t).charCodeAt(0)==43))?1:0,P=D;P<i;P++)if($u((w.Szb(P,w.Xzb(t).length),w.Xzb(t).charCodeAt(P)))==-1)throw Ut(new T.Tib(He+t+'"'));if(u=parseInt(t,10),r=u<e,isNaN(u))throw Ut(new T.Tib(He+t+'"'));if(r||u>A)throw Ut(new T.Tib(He+t+'"'));return u;},T.Rhb=function(t){return gn(t)?T.Vhb(t):t.Zc();},T.Shb=function(t){return gn(t)?T.Zhb(t):t.$c();},x(121,1,{4:1,121:1}),T.Thb=function(t,e){return Ge(w.Zzb((w.Lzb(t),t)),w.Zzb((w.Lzb(e),e)));},T.Uhb=function(t,e){return T.Thb(t,e);},T.Vhb=function(t){return w.Zzb((w.Lzb(t),t));},T.Whb=function(t,e){return w.Lzb(t),k1(t)===k1(e);},T.Xhb=function(){return l.zbb;},T.Yhb=function(t){return Ft(w.Zzb((w.Lzb(t),t)));},T.Zhb=function(t){return Ft(w.Zzb((w.Lzb(t),t)));};function Gu(f){return T.ejb($1,typeof f);}function Ge(f,t){return f<t?-1:f>t?1:f==t?f==0?Ge(1/f,1/t):0:isNaN(f)?isNaN(t)?0:1:-1;}function HP(f){return isNaN(f)?{l:0,m:0,h:524160}:w.Tzb(f);}ZD={4:1,32:1,301:1,121:1},T.FB=function(t){t.g=z(l.Obb,l1,90,0,0,1);},T.GB=function(t){Xt.l5(t);},T.HB=function(t){return Xt.m5(t);},T.IB=function(t){return t.j&&(t.backingJsObject!==VA&&t.ic(),t.g=null),t;},T.JB=function(t){return t.e;},T.KB=function(t,e){if(e instanceof Object)try{if(e.__java$exception=t,dt.userAgent.toLowerCase().indexOf("msie")!=-1&&ft.documentMode<9)return;var A=t;Object.defineProperties(e,{cause:{get:function(){var P=A.gc();return P&&P.ec();}},suppressed:{get:function(){return A.fc();}}});}catch{}},T.LB=function(t,e,A){var P,r,i,D,u;for(T.MB(t),r=(t.i==null&&(t.i=z(l.Tbb,l1,27,0,0,1)),t.i),i=0,D=r.length;i<D;++i)P=r[i],T.LB(P,e,"	"+A);u=t.e,u&&T.LB(u,e,A);},T.MB=function(t){var e,A,P;for(e=(t.g==null&&(t.g=T.HB(t)),t.g),A=0,P=e.length;A<P;++A);},T.NB=function(t,e){t.backingJsObject=e,T.KB(t,e);},T.OB=function(t){return T.PB(t,t.hc());},T.PB=function(t,e){var A;return A=T.qhb(t.ud),e==null?A:A+": "+e;},T.QB=function(){T.FB(this),T.IB(this),this.ic();},T.RB=function(t){T.FB(this),this.f=t,T.IB(this),this.ic();},T.TB=function(t){if(!("stack"in t))try{throw t;}catch{}return t;};function Ku(f){var t;return f!=null&&(t=f.__java$exception,t)?t:YD(f,TypeError)?new T.Qib(f):new T.k4(f);}x(27,1,{4:1,27:1}),y.dc=function(t){return new Error(t);},y.ec=function(){return this.backingJsObject;},y.fc=function(){var t,e,A;for(A=(this.i==null&&(this.i=z(l.Tbb,l1,27,0,0,1)),this.i),e=z(l.Mbb,C1,1,A.length,5,1),t=0;t<A.length;t++)e[t]=A[t].backingJsObject;return e;},y.gc=function(){return T.JB(this);},y.hc=function(){return this.f;},y.ic=function(){T.NB(this,T.TB(this.dc(T.PB(this,this.f)))),T.GB(this);},y.ob=function(){return T.OB(this);},y.backingJsObject=VA,y.j=true,T._B=function(){T.QB.call(this);},T.aC=function(t){T.FB(this),T.IB(this),this.backingJsObject=t,T.KB(this,t),this.f=t==null?re:Zn(t);},T.bC=function(t){T.RB.call(this,t);},x(19,27,{4:1,19:1,27:1},T.bC),T.cC=function(){T._B.call(this);},T.dC=function(t){T.aC.call(this,t);},T.eC=function(t){T.bC.call(this,t);},x(33,19,B0),T.Fgb=function(){T.cC.call(this);},T.Ggb=function(t){T.eC.call(this,t);},x(79,33,B0,T.Fgb,T.Ggb),T.j4=function(){T.cC.call(this);},T.k4=function(t){T.dC.call(this,t);},T.l4=function(t){T.eC.call(this,t);},x(130,33,B0,T.k4),T.Pib=function(){T.j4.call(this);},T.Qib=function(t){T.k4.call(this,t);},T.Rib=function(t){T.l4.call(this,t);},x(69,130,B0,T.Pib,T.Qib,T.Rib),y.dc=function(t){return new TypeError(t);},T.Yib=function(t,e){return w.Szb(e,w.Xzb(t).length),w.Xzb(t).charCodeAt(e);},T.Zib=function(t,e){return T.$ib(t,e);},T.$ib=function(t,e){var A,P;return A=T.Vhb(w.Xzb(t)),P=T.Vhb(w.Xzb(e)),A==P?0:A<P?-1:1;},T._ib=function(t,e){return T.$ib(w.Xzb(t).toLowerCase(),w.Xzb(e).toLowerCase());},T.ajb=function(t,e){return w.Lzb(t),t+(""+(w.Lzb(e),e));},T.bjb=function(t){var e;return T.Bjb(w.vzb(t,0,(e=t.length,w.szb(),e)));},T.cjb=function(t){return T.Bjb(w.vzb(t,0,t.length));},T.djb=function(t){var e;return e=w.Xzb(vn).length,T.ejb(w.Xzb(t).substr(w.Xzb(t).length-e,e),vn);},T.ejb=function(t,e){return w.Lzb(t),k1(t)===k1(e);},T.fjb=function(t,e){return w.Lzb(t),e==null?false:T.ejb(t,e)?true:w.Xzb(t).length==w.Xzb(e).length&&T.ejb(w.Xzb(t).toLowerCase(),w.Xzb(e).toLowerCase());},T.gjb=function(t,e,A,P,r){w.Rzb(e,A,w.Xzb(t).length),w.Rzb(r,r+(A-e),P.length),T.hjb(t,e,A,P,r);},T.hjb=function(t,e,A,P,r){for(;e<A;)P[r++]=T.Yib(t,e++);},T.ijb=function(){return l.Sbb;},T.jjb=function(t){var e,A;for(e=0,A=0;A<w.Xzb(t).length;A++)e=(e<<5)-e+(w.Szb(A,w.Xzb(t).length),w.Xzb(t).charCodeAt(A))|0;return e;},T.kjb=function(t,e,A){return T.mjb(t,Z1(e),A);},T.ljb=function(t,e){return w.Xzb(t).indexOf(e);},T.mjb=function(t,e,A){return w.Xzb(t).indexOf(e,A);};function Yu(f){return T.ejb(M0,typeof f);}T.ojb=function(t,e){return w.Xzb(t).lastIndexOf(e);},T.pjb=function(t,e,A){return w.Xzb(t).lastIndexOf(e,A);},T.qjb=function(t,e){var A;return A=T.rjb(e,"([/\\\\\\.\\*\\+\\?\\|\\(\\)\\[\\]\\{\\}$^])","\\\\$1"),T.rjb(t,A,"");},T.rjb=function(t,e,A){return A=T.Ajb(A),w.Xzb(t).replace(new RegExp(e,"g"),A);},T.sjb=function(t,e){var A,P,r,i,D,u,o,Q;for(A=new RegExp(e,"g"),o=z(l.Sbb,st,2,0,6,1),P=0,Q=t,i=null;;)if(u=A.exec(Q),u==null||Q==""){o[P]=Q;break;}else D=u.index,o[P]=(w.Rzb(0,D,w.Xzb(Q).length),w.Xzb(Q).substr(0,D)),Q=T.vjb(Q,D+w.Xzb(w.Xzb(u)[0]).length,w.Xzb(Q).length),A.lastIndex=0,i==Q&&(o[P]=(w.Rzb(0,1,w.Xzb(Q).length),w.Xzb(Q).substr(0,1)),Q=(w.Szb(1,w.Xzb(Q).length+1),w.Xzb(Q).substr(1))),i=Q,++P;if(w.Xzb(t).length>0){for(r=o.length;r>0&&o[r-1]=="";)--r;r<o.length&&(w.Xzb(o).length=r);}return o;},T.tjb=function(t,e){return T.ejb(w.Xzb(t).substr(0,w.Xzb(e).length),e);},T.ujb=function(t,e){return w.Szb(e,w.Xzb(t).length+1),w.Xzb(t).substr(e);},T.vjb=function(t,e,A){return w.Rzb(e,A,w.Xzb(t).length),w.Xzb(t).substr(e,A-e);},T.wjb=function(t){var e,A,P;for(A=w.Xzb(t).length,P=0;P<A&&(w.Szb(P,w.Xzb(t).length),w.Xzb(t).charCodeAt(P)<=32);)++P;for(e=A;e>P&&(w.Szb(e-1,w.Xzb(t).length),w.Xzb(t).charCodeAt(e-1)<=32);)--e;return P>0||e<A?(w.Rzb(P,e,w.Xzb(t).length),w.Xzb(t).substr(P,e-P)):t;},T.xjb=function(t){return String.fromCharCode.apply(null,t);};function Z1(f){var t,e;return f>=q1?(t=55296+(f-q1>>10&1023)&r1,e=56320+(f-q1&1023)&r1,String.fromCharCode(t)+(""+String.fromCharCode(e))):String.fromCharCode(f&r1);}T.zjb=function(t){var e,A,P,r,i;for(i=new U.Jvb(),A=t,P=0,r=A.length;P<r;++P)e=A[P],i.a?T.Njb(i.a,i.b):i.a=new T.Rjb(i.d),T.Mjb(i.a,e);return i.a?w.Xzb(i.e).length==0?i.a.a:i.a.a+(""+i.e):i.c;},T.Ajb=function(t){var e;for(e=0;0<=(e=w.Xzb(t).indexOf("\\",e));)w.Szb(e+1,w.Xzb(t).length),w.Xzb(t).charCodeAt(e+1)==36?t=(w.Rzb(0,e,w.Xzb(t).length),w.Xzb(t).substr(0,e)+"$"+T.ujb(t,++e)):t=(w.Rzb(0,e,w.Xzb(t).length),w.Xzb(t).substr(0,e)+(""+T.ujb(t,++e)));return t;},T.Bjb=function(t){return T.Cjb(t,0,t.length);},T.Cjb=function(t,e,A){var P,r,i,D;for(i=e+A,w.Rzb(e,i,t.length),D="",r=e;r<i;)P=j.Math.min(r+Y0,i),D+=T.xjb(w.Xzb(t).slice(r,P)),r=P;return D;},_D={4:1,206:1,32:1,2:1},T.Sjb=function(t){T.Ggb.call(this,t);},x(231,79,B0,T.Sjb),w.hzb=function(t){var e;return e=w.Xzb(t).slice(),w.rzb(e,t);},w.izb=function(t,e,A,P,r,i){var D,u,o,Q,L;if(r!=0)for(k1(t)===k1(A)&&(t=w.Xzb(t).slice(e,e+r),e=0),o=w.Xzb(A),u=e,Q=e+r;u<Q;)D=j.Math.min(u+Y0,Q),r=D-u,L=w.Xzb(t).slice(u,D),w.Xzb(L).splice(0,0,w.Xzb(P),w.Xzb(i?r:0)),Array.prototype.splice.apply(o,L),u=D,P+=r;},w.jzb=function(t,e){return w.rzb(new Array(e),t);},w.kzb=function(t,e,A){w.Xzb(t).splice(e,0,A);},w.lzb=function(t,e,A){w.izb(A,0,t,e,A.length,false);},w.mzb=function(t,e){w.Xzb(t).push(e);},w.nzb=function(t,e){w.Xzb(t).push(e);},w.ozb=function(t,e,A){w.Xzb(t).splice(e,A);},w.pzb=function(t,e){w.Xzb(t).length=e;},w.qzb=function(t,e){w.Xzb(t).sort(e);},x(500,1,{}),w.Azb=function(){},w.Bzb=function(t){switch(typeof t){case M0:return T.jjb(w.Xzb(t));case $1:return T.Yhb(w.Xzb(t));case u0:return T.Qgb(w.Xzb(t));default:return t==null?0:w.Dzb(t);}},w.Czb=function(){return ++w.zzb;},w.Dzb=function(t){return t.$H||(t.$H=w.Czb());},x(434,1,{},w.Azb),w.zzb=0,w.Ezb=function(t){if(!t)throw Ut(new T.cib());},w.Fzb=function(t,e){if(!t)throw Ut(new T.dib(e));},w.Gzb=function(t,e,A){if(t>e)throw Ut(new T.dib(tA+t+I8+e));if(t<0||e>A)throw Ut(new T.Igb(tA+t+ZP+e+x8+A));},w.Hzb=function(t){if(t<0)throw Ut(new T.Oib("Negative array size: "+t));},w.Izb=function(t,e){if(t!=e)throw Ut(new U.rtb());},w.Jzb=function(t){if(!t)throw Ut(new U.Lub());},w.Kzb=function(t,e){if(t<0||t>=e)throw Ut(new T.Ggb(_P+t+WP+e));},w.Lzb=function(t){if(t==null)throw Ut(new T.Pib());return t;},w.Mzb=function(t,e){if(t==null)throw Ut(new T.Rib(e));},w.Nzb=function(t,e){if(t<0||t>e)throw Ut(new T.Ggb(_P+t+WP+e));},w.Ozb=function(t,e,A){if(t<0||e>A)throw Ut(new T.Ggb(tA+t+ZP+e+", size: "+A));if(t>e)throw Ut(new T.dib(tA+t+I8+e));},w.Pzb=function(t){if(!t)throw Ut(new T.eib());},w.Qzb=function(t){if(!t)throw Ut(new T.fib("Unable to add element to queue"));},w.Rzb=function(t,e,A){if(t<0||e>A||e<t)throw Ut(new T.Sjb(tA+t+ZP+e+x8+A));},w.Szb=function(t,e){if(t<0||t>=e)throw Ut(new T.Sjb(_P+t+WP+e));},w.Tzb=function(t){var e,A;return e=new ArrayBuffer(8),w.Xzb(new Float64Array(e))[0]=t,A=w.Xzb(new Uint32Array(e)),w.$zb(A[0]|0,A[1]|0);},w.Uzb=function(t,e){return t[e];},w.Vzb=function(t){return t===void 0;},w.Wzb=function(t){return t>>>0;},w.Xzb=function(t){return t;},w.Yzb=function(t){return t;},w.Zzb=function(t){return t;},x(435,1,{}),l.Mbb=X(1),l.Vab=X(0),l.abb=X(436),l.Zab=X(305),l._ab=X(437),l.$ab=X(306),l.wbb=X(300),l.ybb=X(207),l.Lbb=X(121),l.zbb=X(301),l.Tbb=X(27),l.Bbb=X(19),l.Nbb=X(33),l.Ebb=X(79),l.Gbb=X(130),l.Jbb=X(69),l.Sbb=X(2),l.Rbb=X(231),l.Tdb=X(434),l.Udb=X(435),ue.Vc=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t;for(this.c=t.length,this.d=t[0].length,Qt=j.Math.min(this.c,this.d),this.e=z(l.b7,jt,6,j.Math.min(this.c+1,this.d),15,1),this.a=h1(l.b7,[l1,jt],[15,6],15,[this.c,Qt],2),this.b=h1(l.b7,[l1,jt],[15,6],15,[this.d,this.d],2),r=z(l.b7,jt,6,this.d,15,1),$t=z(l.b7,jt,6,this.c,15,1),I=j.Math.min(this.c-1,this.d),ut=j.Math.max(0,j.Math.min(this.d-2,this.c)),S=0;S<j.Math.max(I,ut);S++){if(S<I){for(this.e[S]=0,s=S;s<this.c;s++)this.e[S]=ue.Wc(this.e[S],t[s][S]);if(this.e[S]!=0){for(t[S][S]<0&&(this.e[S]=-this.e[S]),L=S;L<this.c;L++)t[L][S]/=this.e[S];t[S][S]+=1;}this.e[S]=-this.e[S];}for(d=S+1;d<this.d;d++){if(S<I&this.e[S]!=0){for(wt=0,s=S;s<this.c;s++)wt+=t[s][S]*t[s][d];for(wt=-wt/t[S][S],L=S;L<this.c;L++)t[L][d]+=wt*t[L][S];}r[d]=t[S][d];}if(true&S<I)for(L=S;L<this.c;L++)this.a[L][S]=t[L][S];if(S<ut){for(r[S]=0,s=S+1;s<this.d;s++)r[S]=ue.Wc(r[S],r[s]);if(r[S]!=0){for(r[S+1]<0&&(r[S]=-r[S]),h=S+1;h<this.d;h++)r[h]/=r[S];r[S+1]+=1;}if(r[S]=-r[S],S+1<this.c&r[S]!=0){for(h=S+1;h<this.c;h++)$t[h]=0;for(m=S+1;m<this.d;m++)for(b=S+1;b<this.c;b++)$t[b]+=r[m]*t[b][m];for(C=S+1;C<this.d;C++)for(wt=-r[C]/r[S+1],b=S+1;b<this.c;b++)t[b][C]+=wt*$t[b];}for(L=S+1;L<this.d;L++)this.b[L][S]=r[L];}}for(Dt=j.Math.min(this.d,this.c+1),I<this.d&&(this.e[I]=t[I][I]),this.c<Dt&&(this.e[Dt-1]=0),ut+1<Dt&&(r[ut]=t[ut][Dt-1]),r[Dt-1]=0,v=I;v<Qt;v++){for(L=0;L<this.c;L++)this.a[L][v]=0;this.a[v][v]=1;}for(V=I-1;V>=0;V--)if(this.e[V]!=0){for(g=V+1;g<Qt;g++){for(wt=0,s=V;s<this.c;s++)wt+=this.a[s][V]*this.a[s][g];for(wt=-wt/this.a[V][V],h=V;h<this.c;h++)this.a[h][g]+=wt*this.a[h][V];}for(b=V;b<this.c;b++)this.a[b][V]=-this.a[b][V];for(this.a[V][V]=1+this.a[V][V],L=0;L<V-1;L++)this.a[L][V]=0;}else {for(L=0;L<this.c;L++)this.a[L][V]=0;this.a[V][V]=1;}for(O=this.d-1;O>=0;O--){if(O<ut&r[O]!=0)for(g=O+1;g<Qt;g++){for(wt=0,s=O+1;s<this.d;s++)wt+=this.b[s][O]*this.b[s][g];for(wt=-wt/this.b[O+1][O],h=O+1;h<this.d;h++)this.b[h][g]+=wt*this.b[h][O];}for(L=0;L<this.d;L++)this.b[L][O]=0;this.b[O][O]=1;}for(Lt=Dt-1,u=j.Math.pow(2,-52);Dt>0;){for(M=Dt-2;M>=-1&&M!=-1;M--)if(j.Math.abs(r[M])<=u*(j.Math.abs(this.e[M])+j.Math.abs(this.e[M+1]))){r[M]=0;break;}if(M==Dt-2)K=4;else {for(N=Dt-1;N>=M&&N!=M;N--)if(wt=(N!=Dt?j.Math.abs(r[N]):0)+(N!=M+1?j.Math.abs(r[N-1]):0),j.Math.abs(this.e[N])<=u*wt){this.e[N]=0;break;}N==M?K=3:N==Dt-1?K=1:(K=2,M=N);}switch(++M,K){case 1:for(o=r[Dt-2],r[Dt-2]=0,g=Dt-2;g>=M;g--)for(wt=ue.Wc(this.e[g],o),P=this.e[g]/wt,bt=o/wt,this.e[g]=wt,g!=M&&(o=-bt*r[g-1],r[g-1]=P*r[g-1]),L=0;L<this.d;L++)wt=P*this.b[L][g]+bt*this.b[L][Dt-1],this.b[L][Dt-1]=-bt*this.b[L][g]+P*this.b[L][Dt-1],this.b[L][g]=wt;break;case 2:for(o=r[M-1],r[M-1]=0,g=M;g<Dt;g++)for(wt=ue.Wc(this.e[g],o),P=this.e[g]/wt,bt=o/wt,this.e[g]=wt,o=-bt*r[g],r[g]=P*r[g],L=0;L<this.c;L++)wt=P*this.a[L][g]+bt*this.a[L][M-1],this.a[L][M-1]=-bt*this.a[L][g]+P*this.a[L][M-1],this.a[L][g]=wt;break;case 3:{for(ot=j.Math.max(j.Math.max(j.Math.max(j.Math.max(j.Math.abs(this.e[Dt-1]),j.Math.abs(this.e[Dt-2])),j.Math.abs(r[Dt-2])),j.Math.abs(this.e[M])),j.Math.abs(r[M])),it=this.e[Dt-1]/ot,ct=this.e[Dt-2]/ot,D=r[Dt-2]/ot,yt=this.e[M]/ot,i=r[M]/ot,e=((ct+it)*(ct-it)+D*D)/2,A=it*D*(it*D),rt=0,e!=0|A!=0&&(rt=j.Math.sqrt(e*e+A),e<0&&(rt=-rt),rt=A/(e+rt)),o=(yt+it)*(yt-it)+rt,Q=yt*i,g=M;g<Dt-1;g++){for(wt=ue.Wc(o,Q),P=o/wt,bt=Q/wt,g!=M&&(r[g-1]=wt),o=P*this.e[g]+bt*r[g],r[g]=P*r[g]-bt*this.e[g],Q=bt*this.e[g+1],this.e[g+1]=P*this.e[g+1],s=0;s<this.d;s++)wt=P*this.b[s][g]+bt*this.b[s][g+1],this.b[s][g+1]=-bt*this.b[s][g]+P*this.b[s][g+1],this.b[s][g]=wt;if(wt=ue.Wc(o,Q),P=o/wt,bt=Q/wt,this.e[g]=wt,o=P*r[g]+bt*this.e[g+1],this.e[g+1]=-bt*r[g]+P*this.e[g+1],Q=bt*r[g+1],r[g+1]=P*r[g+1],g<this.c-1)for(L=0;L<this.c;L++)wt=P*this.a[L][g]+bt*this.a[L][g+1],this.a[L][g+1]=-bt*this.a[L][g]+P*this.a[L][g+1],this.a[L][g]=wt;}r[Dt-2]=o;}break;case 4:{if(this.e[M]<=0)for(this.e[M]=this.e[M]<0?-this.e[M]:0,L=0;L<=Lt;L++)this.b[L][M]=-this.b[L][M];for(;M<Lt&&!(this.e[M]>=this.e[M+1]);){if(wt=this.e[M],this.e[M]=this.e[M+1],this.e[M+1]=wt,M<this.d-1)for(L=0;L<this.d;L++)wt=this.b[L][M+1],this.b[L][M+1]=this.b[L][M],this.b[L][M]=wt;if(M<this.c-1)for(L=0;L<this.c;L++)wt=this.a[L][M+1],this.a[L][M+1]=this.a[L][M],this.a[L][M]=wt;++M;}--Dt;}}}},ue.Wc=function(t,e){var A;return j.Math.abs(t)>j.Math.abs(e)?(A=e/t,A=j.Math.abs(t)*j.Math.sqrt(1+A*A)):e!=0?(A=t/e,A=j.Math.abs(e)*j.Math.sqrt(1+A*A)):A=0,A;},x(200,1,fn,ue.Vc),y.c=0,y.d=0,l.f7=X(200),n._c=function(){n._c=o1,n.Xc=F(k(l.d7,1),H,6,15,[0,We,14286847,13402367,12779264,16758197,9474192,3166456,16715021,9494608,11789301,11230450,9109248,12560038,1578e4,16744448,16777008,2093087,8442339,9388244,4062976,15132390,12567239,10921643,9083335,10255047,14706227,15765664,5296208,13140019,8224944,12750735,6721423,12419299,16752896,10889513,6076625,7351984,J0,9764863,9756896,7586505,5551541,3907230,2396047,687500,27013,12632256,16767375,10909043,6717568,10380213,13924864,9699476,4366e3,5707663,51456,7394559,16777159,14286791,13107143,10747847,9437127,6422471,4587463,3211207,2097095,65436,58997,54354,48952,43812,5096191,5089023,2200790,2522539,2516630,1528967,13684960,16765219,12105936,10900557,5724513,10375093,11230208,7688005,4358806,4325478,32e3,7384058,47871,41471,36863,33023,27647,5528818,7888099,9064419,10565332,11739092,11739066,11734438,12389767,13041766,13369433,13697103,14221381,14680120,15073326,15400998,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13158600,1334015,56540,15075850,15132160,56540,15075850,15461355,8553170,1016335,1016335,1334015,15132160,3289770,14456450,16422400,16422400,11819700,3289770,1016335]);},n.ad=function(t){t.v=new Tt.EN();},n.bd=function(t){var e,A;(t.G&32)==0&&(A=n.dw(t.L),A!=null&&(t.w.a==0&&t.w.b==0&&(e=t.R.c*n.Tl(t.L),t.v=n.Zd(t),n.od(t,e),n.Ud(t,null,e,0)),t.Bb(Ft(t.A)),t.L.O!=1&&n.Vd(t,448),t.sb(A,t.w.a,t.w.b+Dn*t.A)));},n.cd=function(t,e){return t==null?e:e==null?t:t+","+e;},n.dd=function(t,e){n.ti(e,t.R),n.wi(e,t.v),n.vi(e,t.w);},n.ed=function(t){var e;e=t.R.c*(t.r!=0?t.r:n.Tl(t.L)),t.Y=e*n.Zc,t.T=e*n.Yc,t.S=e*.38,t.W=e*.47,t.X=Ft(e*t.J*n.$c+.5),t.V=e*.12,t.Z=e*.4,t.A=e*.5+.5;},n.fd=function(t){var e,A;for(t=G(t,Me),A=0;A<(q.rK(),q.qK).length;A++)if(M1(t,q.qK[A]))return q.pK[A];return e=new T.Rjb("R"),kt(G(t,Qe),0)&&(e.a+="0"),kt(G(t,Se),0)&&(e.a+="3"),kt(G(t,qe),0)&&(e.a+="4"),kt(G(t,Ve),0)&&(e.a+="5"),kt(G(t,Ee),0)&&(e.a+="6"),kt(G(t,Ne),0)&&(e.a+="7"),kt(G(t,un),0)&&(e.a+="8"),e.a;},n.gd=function(t,e,A,P){var r,i;r=new n.je(),i=new n.je(),r.a=e.a,r.c=e.c,r.b=(e.a+e.b)/2,r.d=(e.c+e.d)/2,i.a=r.b,i.c=r.d,i.b=e.b,i.d=e.d,n.Rd(t,r)&&(n.Vd(t,t.p[A]),t.pb(r)),n.Rd(t,i)&&(n.Vd(t,t.p[P]),t.pb(i)),n.Vd(t,t.Q);},n.hd=function(t,e,A,P){var r,i,D,u,o;u=(e.b-e.a)/10,o=(e.d-e.c)/10,r=new n.je(),n.Cm(t.L,n.Wo(t.L,A,P))?(i=-3,D=-3):(i=t.p[A],D=t.p[P]),n.Vd(t,i),r.a=e.a,r.c=e.c,r.b=e.a+u*2,r.d=e.c+o*2,t.pb(r),r.a=e.a+u*4,r.c=e.c+o*4,r.b=e.a+u*5,r.d=e.c+o*5,t.pb(r),n.Vd(t,D),r.a=e.a+u*5,r.c=e.c+o*5,r.b=e.a+u*6,r.d=e.c+o*6,t.pb(r),r.a=e.a+u*8,r.c=e.c+o*8,r.b=e.b,r.d=e.d,t.pb(r),n.Vd(t,t.Q);},n.jd=function(t,e,A){t.tb(e-t.V/2,A-t.V/2,t.V);},n.kd=function(t,e,A,P){n.Cm(t.L,n.Wo(t.L,A,P))?(n.Vd(t,-3),t.pb(e),n.Vd(t,t.Q)):t.p[A]!=t.p[P]?n.gd(t,e,A,P):t.p[A]!=0?(n.Vd(t,t.p[A]),t.pb(e),n.Vd(t,t.Q)):t.pb(e);},n.ld=function(t,e,A,P,r,i,D){D.a=e.a+r,D.c=e.c+i,D.b=e.b+r,D.d=e.d+i,n.kd(t,D,A,P);},n.md=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g;for(s=e.b-e.a,c=e.d-e.c,o=j.Math.sqrt(s*s+c*c),Q=2*Gt(w1(j.Math.round(o/(4*t.Y)))),h=s/(Q-1),g=c/(Q-1),n.Cm(t.L,n.Wo(t.L,A,P))?(r=-3,i=-3):(r=t.p[A],i=t.p[P]),L=e.a-t.Y/2,b=e.c-t.Y/2,n.Vd(t,r),u=0;u<(Q/2|0);u++)t.tb(L,b,t.Y),L+=h,b+=g;for(n.Vd(t,i),D=0;D<(Q/2|0);D++)t.tb(L,b,t.Y),L+=h,b+=g;n.Vd(t,t.Q);},n.nd=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m;o=(e.c-e.d)/9,b=(e.b-e.a)/9,Q=e.b+o,c=e.d+b,L=e.b-o,g=e.d-b,s=(e.a+Q)/2,d=(e.c+c)/2,h=(e.a+L)/2,m=(e.c+g)/2,D=new Tt.yN(3),u=new Tt.yN(4),Tt.tN(D,e.a,e.c),Tt.tN(D,s,d),Tt.tN(D,h,m),Tt.tN(u,h,m),Tt.tN(u,s,d),Tt.tN(u,Q,c),Tt.tN(u,L,g),n.Cm(t.L,n.Wo(t.L,A,P))?(r=-3,i=-3):(r=t.p[A],i=t.p[P],t.L.O!=1&&(i=n.qd(t,A),r==n.Bl(t.L,A)&&(r=i))),n.Vd(t,r),t.rb(D),n.Vd(t,i),t.rb(u),n.Vd(t,t.Q);},n.od=function(t,e){var A,P;for(P=0;P<t.$.a.length;P++)t.v=Tt.DN(t.v,U.wj(t.$,P));n.pd(t,e),A=.1*e,t.v.c-=A,t.v.d-=A,t.v.b+=2*A,t.v.a+=2*A;},n.pd=function(t,e){var A,P,r,i,D,u,o;for(r=z(l.zeb,ht,6,t.L.q,16,1),P=0;P<t.L.r;P++)n.Am(t.L,P)&&(r[n.Yl(t.L,0,P)]=true,r[n.Yl(t.L,1,P)]=true);for(D=new Tt.EN(),A=0;A<t.L.q;A++)i=n.Im(t.L,A)?e*.47:r[A]?e*.38:0,i!=0&&(u=n.zi(t.R,n.Pl(t.L,A)),o=n.Ai(t.R,n.Ql(t.L,A)),Tt.CN(D,u-i,o-i,i*2,i*2),t.v=Tt.DN(t.v,D));},n.qd=function(t,e){var A,P;if((t.G&4224)!=0||(P=n.rd(t,e),P==-1&&(A=n.Eo(t.L,e),A!=-1&&(e=A,P=n.rd(t,e))),P==-1))return t.p[e];switch(P&255){case 1:return 384;case 2:return 64;default:return 448;}},n.rd=function(t,e){var A,P,r;return r=-1,P=-1,(t.G&128)!=0||(n.ym(t.L,e)&&(r=n.Gl(t.L,e),P=n.Fl(t.L,e)),A=n.Go(t.L,e),A!=-1&&(r=n.bm(t.L,A),P=n.am(t.L,A)),r!=-1&&r!=0&&(r|=P<<8)),r;},n.sd=function(t){var e,A,P,r;for(t.zb(2*t.S),r=new n.je(),P=0;P<t.L.r;P++)e=n.Yl(t.L,0,P),A=n.Yl(t.L,1,P),n.Am(t.L,P)&&(r.a=n.zi(t.R,n.Pl(t.L,e)),r.c=n.Ai(t.R,n.Ql(t.L,e)),r.b=n.zi(t.R,n.Pl(t.L,A)),r.d=n.Ai(t.R,n.Ql(t.L,A)),n.Vd(t,-2),t.pb(r));},n.td=function(t){var e,A,P,r,i,D;if(t.L.K){for(D=t.W,n.Vd(t,-7),e=0;e<t.L.f;e++)n.Im(t.L,e)&&t.tb(n.zi(t.R,n.Pl(t.L,e))-D,n.Ai(t.R,n.Ql(t.L,e))-D,2*D);for(t.zb(2*t.W),i=new n.je(),r=0;r<t.L.r;r++)A=n.Yl(t.L,0,r),P=n.Yl(t.L,1,r),kt(G(G(n.Nl(t.L,A),n.Nl(t.L,P)),r0),0)&&(i.a=n.zi(t.R,n.Pl(t.L,A)),i.c=n.Ai(t.R,n.Ql(t.L,A)),i.b=n.zi(t.R,n.Pl(t.L,P)),i.d=n.Ai(t.R,n.Ql(t.L,P)),t.pb(i));}},n.ud=function(t){var e,A,P,r;if(t.L.K){if(n.Vd(t,320),(t.G&8)!=0)for(e=0;e<t.L.f;e++)kt(G(n.Nl(t.L,e),-536870913),0)&&t.tb(n.zi(t.R,n.Pl(t.L,e))-t.Z/2,n.Ai(t.R,n.Ql(t.L,e))-t.Z/2,t.Z);for(r=0;r<t.L.g;r++)n.gm(t.L,r)!=0&&(A=n.Yl(t.L,0,r),P=n.Yl(t.L,1,r),t.tb((n.zi(t.R,n.Pl(t.L,A))+n.zi(t.R,n.Pl(t.L,P))-t.Z)/2,(n.Ai(t.R,n.Ql(t.L,A))+n.Ai(t.R,n.Ql(t.L,P))-t.Z)/2,t.Z));}},n.vd=function(t){t.J=1,t.R=new n.Bi(),t.$=new U.Lj(),t.U=new U.Lj(),t.s=z(l.zeb,ht,6,t.L.q,16,1),t.w=new Tt.qN(),t.Q=0,t.C=-1,n._d(t);},n.wd=function(t,e){var A;if(n.Ko(t.L,e)==0)return  false;for(A=0;A<n.Ko(t.L,e);A++)if(!n.Cm(t.L,n.$o(t.L,e,A)))return  false;return  true;},n.xd=function(t){var e;for(t.q=z(l.zeb,ht,6,t.L.q,16,1),e=0;e<t.L.r;e++)t.q[n.Yl(t.L,0,e)]=true,t.q[n.Yl(t.L,1,e)]=true;},n.yd=function(t,e){var A;if(n.Zo(t.L,e)!=2)return  false;for(A=0;A<2;A++)if(n._o(t.L,e,A)!=2)return  false;return  true;},n.zd=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c;for(h=false,r.a=0,r.b=0,P>0?i=$P:i=aP,c=n.Xl(t.L,e,A),L=0;L<n.Zo(t.L,e);L++)D=n.$o(t.L,e,L),u=c,n.Yl(t.L,0,D)==e?s=n.Yl(t.L,1,D):s=n.Yl(t.L,0,D),s!=A&&(b=n.Xl(t.L,e,s),u<b&&(u+=X1),o=u-b,P>0?(o<n1&&(h=true),o>$P&&(o=$P),o<.523598776&&(o=.523598776),o<=i&&(i=o,Q=t.T*j.Math.tan(i-m0)/2,r.a=-(Q*j.Math.sin(u)),r.b=-(Q*j.Math.cos(u)))):(o>=n1&&(h=true),o<aP&&(o=aP),o>5.759586531&&(o=5.759586531),o>=i&&(i=o,Q=t.T*j.Math.tan(4.712388981-i)/2,r.a=-(Q*j.Math.sin(u)),r.b=-(Q*j.Math.cos(u)))));return h;},n.Ad=function(t,e,A,P){var r;if(e==0){A<0?P.a=t.T:P.a=-t.T,P.b=0;return;}r=j.Math.atan(A/e),e<0&&(r+=n1),P.a=-(t.T*j.Math.sin(r)),P.b=t.T*j.Math.cos(r);},n.Bd=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g;r=new n.je(),o=new n.je(),L=new Tt.qN(),Q=new Tt.qN(),D=n.Yl(t.L,0,A),u=n.Yl(t.L,1,A),P&&(h=e.a,e.a=e.b,e.b=h,h=e.c,e.c=e.d,e.d=h,b=D,D=u,u=b),n.Rd(t,e)&&(n.fq(t.L,A)?(r.a=e.a,r.c=e.c,r.b=e.b,r.d=e.d,s=P?-n.Cp(t.L,A):n.Cp(t.L,A),s==0&&(s=1),n.Ad(t,e.b-e.a,e.d-e.c,L),s>0?(o.a=e.a+L.a,o.c=e.c+L.b,o.b=e.b+L.a,o.d=e.d+L.b,(n.zd(t,D,u,1,Q)||n.Zo(t.L,D)>1)&&(o.a+=Q.a+L.b,o.c+=Q.b-L.a)):(o.a=e.a-L.a,o.c=e.c-L.b,o.b=e.b-L.a,o.d=e.d-L.b,(n.zd(t,D,u,-1,Q)||n.Zo(t.L,D)>1)&&(o.a+=Q.a+L.b,o.c+=Q.b-L.a)),n.hm(t.L,A)==386&&n.Qd(r,o),n.Rd(t,r)&&n.kd(t,r,D,u),n.hm(t.L,A)==8?n.Rd(t,o)&&n.hd(t,o,D,u):n.Rd(t,o)&&n.kd(t,o,D,u)):(n.Ad(t,e.b-e.a,e.d-e.c,L),c=L.a/2,g=L.b/2,i=false,r.a=e.a+c,r.c=e.c+g,r.b=e.b+c,r.d=e.d+g,n.Zo(t.L,D)>1&&(n.zd(t,D,u,1,Q)?(r.a+=Q.a,r.c+=Q.b,n.Zo(t.L,D)==2&&(Q.a!=0||Q.b!=0)&&(r.a+=L.b,r.c-=L.a)):t.o[D]=new Tt.rN(r.a,r.c)),o.a=e.a-c,o.c=e.c-g,o.b=e.b-c,o.d=e.d-g,n.Zo(t.L,D)>1&&(n.zd(t,D,u,0,Q)?(o.a+=Q.a,o.c+=Q.b,n.Zo(t.L,D)==2&&(Q.a!=0||Q.b!=0)&&(o.a+=L.b,o.c-=L.a)):(t.o[D]=new Tt.rN(o.a,o.c),i=true)),n.hm(t.L,A)==386&&n.Qd(r,o),n.hm(t.L,A)==8?i?(n.hd(t,r,D,u),n.kd(t,o,D,u)):(n.kd(t,r,D,u),n.hd(t,o,D,u)):(n.kd(t,r,D,u),n.kd(t,o,D,u))));},n.Cd=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c;for(L=t.Q,s=t.F,(t.G&x0)!=0&&(t.Q=-6,t.F=EA,n.Vd(t,1)),t.o=z(l._9,C1,55,t.L.q,0,1),o=0;o<t.L.r;o++)(n.hm(t.L,o)==2||n.hm(t.L,o)==386||n.hm(t.L,o)==8)&&n.Fd(t,o);for(Q=0;Q<t.L.r;Q++)n.hm(t.L,Q)!=2&&n.hm(t.L,Q)!=386&&n.hm(t.L,Q)!=8&&n.Fd(t,Q);if((t.G&64)==0)for(u=0;u<t.L.r;u++)n._l(t.L,u)!=0&&(r=null,n._l(t.L,u)==1||n._l(t.L,u)==2?(n.em(t.L,u)==2||n.bm(t.L,u)==0||e[n.bm(t.L,u)][n.am(t.L,u)]>1)&&(n._l(t.L,u)==1?r=n.em(t.L,u)==2?"E":n.Em(t.L,u)?"p":"P":r=n.em(t.L,u)==2?"Z":n.Em(t.L,u)?"m":"M"):r="?",r!=null&&(t.Bb((t.X*2+1)/3|0),n.Vd(t,n.Cm(t.L,u)?-3:t.L.O==1||(t.G&Y1)!=0?t.Q:448),A=n.Yl(t.L,0,u),P=n.Yl(t.L,1,u),b=(n.zi(t.R,n.Pl(t.L,A))+n.zi(t.R,n.Pl(t.L,P)))/2,c=(n.Ai(t.R,n.Ql(t.L,A))+n.Ai(t.R,n.Ql(t.L,P)))/2,i=(n.zi(t.R,n.Pl(t.L,A))-n.zi(t.R,n.Pl(t.L,P)))/3,D=(n.Ai(t.R,n.Ql(t.L,A))-n.Ai(t.R,n.Ql(t.L,P)))/3,n.Id(t,b+D,c-i,r,true),n.Vd(t,t.Q),t.Bb(t.X)));if((t.G&4)!=0){for(t.Bb((t.X*2+1)/3|0),n.Vd(t,384),u=0;u<t.L.r;u++)A=n.Yl(t.L,0,u),P=n.Yl(t.L,1,u),h=n.Xp(t.L,u)?"d":n.Tp(t.L,u)?"a":"",b=(n.zi(t.R,n.Pl(t.L,A))+n.zi(t.R,n.Pl(t.L,P)))/2,c=(n.Ai(t.R,n.Ql(t.L,A))+n.Ai(t.R,n.Ql(t.L,P)))/2,n.Id(t,b,c,h+(""+u),true);n.Vd(t,t.Q),t.Bb(t.X);}(t.G&x0)!=0&&(t.Q=L,t.F=s);},n.Dd=function(t){var e,A;for(A=new U.wqb(t.U);A.a<A.c.a.length;)e=U.vqb(A),n.Vd(t,e.a),n.jd(t,e.b,e.c);n.Vd(t,t.Q);},n.Ed=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt;if(t.K||t.xb(e,n.Hl(t.L,e),n.zi(t.R,n.Pl(t.L,e)),n.Ai(t.R,n.Ql(t.L,e))),bt=null,n.Al(t.L,e)!=0&&(Ht=j.Math.abs(n.Al(t.L,e))==1?"":""+j.Math.abs(n.Al(t.L,e)),bt=n.Al(t.L,e)<0?Ht+"-":Ht+"+"),I=null,it=n.Nl(t.L,e),et(it,0)!=0&&(kt(G(it,eA),0)&&(I=I==null?"*":I+",*"),kt(G(it,nA),0)&&(I=I==null?"!*":I+",!*"),kt(G(it,AA),0)?I=I==null?"ha":I+",ha":kt(G(it,2),0)?I=I==null?"a":I+",a":kt(G(it,4),0)&&(I=I==null?"!a":I+",!a"),kt(G(it,Y1),0)&&(I=I==null?"s":I+",s"),kt(G(it,Z0),0)&&(V=G(it,Z0),et(V,1792)==0?I=I==null?"h0":I+",h0":et(V,1664)==0?I=I==null?"h1":I+",h1":et(V,1408)==0?I=I==null?"h2":I+",h2":et(V,128)==0?I=I==null?"h>0":I+",h>0":et(V,384)==0?I=I==null?"h>1":I+",h>1":et(V,896)==0?I=I==null?"h>2":I+",h>2":et(V,i0)==0?I=I==null?"h<3":I+",h<3":et(V,1536)==0?I=I==null?"h<2":I+",h<2":et(V,1152)==0&&(I=I==null?"h1-2":I+",h1-2")),kt(G(it,je),0)&&(o=G(it,je),et(o,PA)==0?I=I==null?"c0":I+",c0":et(o,on)==0?I=I==null?"c+":I+",c+":et(o,ln)==0&&(I=I==null?"c-":I+",c-")),kt(G(it,$e),0)&&(yt=G(it,$e),et(yt,98304)==0?I=I==null?"pi0":I+",pi0":et(yt,81920)==0?I=I==null?"pi1":I+",pi1":et(yt,49152)==0?I=I==null?"pi2":I+",pi2":et(yt,x0)==0&&(I=I==null?"pi>0":I+",pi>0")),kt(G(it,ae),0)&&(rt=G(it,ae),et(rt,rA)==0?I=I==null?"n1":I+",n1":et(rt,pn)==0?I=I==null?"n2":I+",n2":et(rt,Rn)==0?I=I==null?"n3":I+",n3":et(rt,iA)==0?I=I==null?"n<3":I+",n<3":et(rt,F0)==0?I=I==null?"n<4":I+",n<4":et(rt,tn)==0?I=I==null?"n>1":I+",n>1":et(rt,ie)==0?I=I==null?"n>2":I+",n>2":et(rt,Qn)==0?I=I==null?"n>3":I+",n>3":et(rt,3276800)==0?I=I==null?"n1-2":I+",n1-2":et(rt,2228224)==0?I=I==null?"n1-3":I+",n1-3":et(rt,2490368)==0&&(I=I==null?"n2-3":I+",n2-3")),kt(G(it,Ln),0)&&(c=G(it,Ln),et(c,NA)==0?I=I==null?"e0":I+",e0":et(c,IA)==0?I=I==null?"e1":I+",e1":et(c,xA)==0?I=I==null?"e2":I+",e2":et(c,OA)==0?I=I==null?"e3":I+",e3":et(c,tr)==0?I=I==null?"e<2":I+",e<2":et(c,er)==0?I=I==null?"e<3":I+",e<3":et(c,fA)==0?I=I==null?"e<4":I+",e<4":et(c,Cn)==0?I=I==null?"e>0":I+",e>0":et(c,XA)==0?I=I==null?"e>1":I+",e>1":et(c,GA)==0?I=I==null?"e>2":I+",e>2":et(c,DA)==0?I=I==null?"e>3":I+",e>3":et(c,nr)==0?I=I==null?"e1-2":I+",e1-2":et(c,Ar)==0?I=I==null?"e1-3":I+",e1-3":et(c,Pr)==0&&(I=I==null?"e2-3":I+",e2-3")),kt(G(it,120),0)&&(ct=G(it,120),et(ct,112)==0?I=I==null?"!r":I+",!r":et(ct,8)==0?I=I==null?"r":I+",r":et(ct,96)==0?I=I==null?"rb<3":I+",rb<3":et(ct,104)==0?I=I==null?"rb2":I+",rb2":et(ct,88)==0?I=I==null?"rb3":I+",rb3":et(ct,56)==0&&(I=I==null?"rb4":I+",rb4")),kt(G(it,Be),0)&&(I=I==null?"r"+y0(t0(G(it,Be),22)):I+","+("r"+y0(t0(G(it,Be),22)))),kt(G(it,Me),0)&&(I=n.cd(I,n.fd(it))),kt(G(it,Ie),0)&&(I=I==null?"f":I+",f")),n.Ll(t.L,e)!=0&&(I=n.cd(I,""+n.Ll(t.L,e))),vt=0,n.Ol(t.L,e)!=0)switch(n.Ol(t.L,e)){case 16:bt=bt==null?"|":bt+",|";break;case 32:vt=1;break;case 48:vt=2;}if(s=null,(t.G&64)==0){if(n.vm(t.L,e))s="?";else if(n.zl(t.L,e)!=0&&(n.Gl(t.L,e)==0||A==null||A[n.Gl(t.L,e)][n.Fl(t.L,e)]>1))if(n.Zo(t.L,e)==2)switch(n.zl(t.L,e)){case 2:s=n.xm(t.L,e)?"p":"P";break;case 1:s=n.xm(t.L,e)?"m":"M";break;default:s="*";}else switch(n.zl(t.L,e)){case 1:s=n.xm(t.L,e)?"r":"R";break;case 2:s=n.xm(t.L,e)?"s":"S";break;default:s="*";}}if((t.G&768)!=0&&(s=n.cd(s,""+n.lw(t.L,e))),Lt=null,(t.G&16)!=0&&n.Kl(t.L,e)!=0&&(Lt=""+n.Kl(t.L,e)),d=null,n.Hp(t.L,e)!=-1&&(g=n.rd(t,e),g!=-1&&(d=g==0?"abs":((g&255)==1?"&":"or")+(1+(g>>8)))),O=0,(t.G&Q0)==0&&(t.L.K?kt(G(n.Nl(t.L,e),z1),0)&&(O=n.np(t.L,e)):(n.Sl(t.L,e)!=6||n.Ll(t.L,e)!=0||!t.q[e]||n.Ol(t.L,e)!=0)&&(O=n.np(t.L,e))),Dt=false,i=n.Dl(t.L,e),i!=null&&T.ejb(w.Xzb(i).substr(0,1),"]")&&(I=n.cd((w.Szb(1,w.Xzb(i).length+1),w.Xzb(i).substr(1)),I),i=null,Dt=true),i!=null?O=0:n.Il(t.L,e)!=null?(r=kt(G(n.Nl(t.L,e),1),0)?"[!":"[",i=r+n.Jl(t.L,e)+"]",w.Xzb(i).length>5&&(i=r+n.Il(t.L,e).length+"]"),kt(G(n.Nl(t.L,e),z1),0)&&(O=-1)):kt(G(n.Nl(t.L,e),1),0)?(i="?",kt(G(n.Nl(t.L,e),z1),0)&&(O=-1)):(n.Sl(t.L,e)!=6||bt!=null||I!=null||O>0||!t.q[e])&&(i=n.Hl(t.L,e)),Qt=0,!n.Om(t.L,e)&n.Im(t.L,e)&&n.Vd(t,-8),i!=null?(Qt=t.ub(i),n.Id(t,n.zi(t.R,n.Pl(t.L,e)),n.Ai(t.R,n.Ql(t.L,e)),i,true),t.s[e]=true):n.yd(t,e)&&n.Hd(t,n.zi(t.R,n.Pl(t.L,e)),n.Ai(t.R,n.Ql(t.L,e)),e),bt!=null&&(t.Bb((t.X*2+1)/3|0),xt=n.zi(t.R,n.Pl(t.L,e))+((Qt+t.ub(bt))/2+1),Nt=n.Ai(t.R,n.Ql(t.L,e))-((t.vb()*4-4)/8|0),n.Id(t,xt,Nt,bt,true),t.Bb(t.X)),(t.G&2)!=0&&(I=""+e),I!=null&&(Dt?t.Bb((t.X*5+1)/6|0):t.Bb((t.X*2+1)/3|0),xt=n.zi(t.R,n.Pl(t.L,e))-(Qt+t.ub(I))/2,Nt=n.Ai(t.R,n.Ql(t.L,e))-((t.vb()*4-4)/8|0),n.Id(t,xt,Nt,I,true),t.Bb(t.X)),s!=null&&(t.Bb((t.X*2+1)/3|0),xt=n.zi(t.R,n.Pl(t.L,e))-(Qt+t.ub(s))/2,Nt=n.Ai(t.R,n.Ql(t.L,e))+((t.vb()*4+4)/8|0),$t=t.C,t.L.O!=1&&(t.G&Y1)==0&&n.Vd(t,448),n.Id(t,xt,Nt,s,false),n.Vd(t,$t),t.Bb(t.X)),Lt!=null&&(t.Bb((t.X*2+1)/3|0),xt=n.zi(t.R,n.Pl(t.L,e))+((Qt+t.ub(Lt))/2+1),Nt=n.Ai(t.R,n.Ql(t.L,e))+((t.vb()*4+4)/8|0),$t=t.C,n.Vd(t,n.zm(t.L,e)?384:448),n.Id(t,xt,Nt,Lt,true),n.Vd(t,$t),t.Bb(t.X)),d!=null&&(P=n.Md(t,e),t.Bb((t.X*2+1)/3|0),xt=n.zi(t.R,n.Pl(t.L,e))+.7*t.vb()*j.Math.sin(P),Nt=n.Ai(t.R,n.Ql(t.L,e))+.7*t.vb()*j.Math.cos(P),$t=t.C,!t.K&&t.L.O!=1&&n.Vd(t,n.qd(t,e)),n.Id(t,xt,Nt,d,false),n.Vd(t,$t),t.Bb(t.X)),O==0&&vt==0){t.C==-8&&n.Vd(t,-9);return;}for(M=z(l.b7,jt,6,4,15,1),N=0;N<n.Lo(t.L,e);N++)for(u=n.$o(t.L,e,N),ut=0;ut<2;ut++)n.Yl(t.L,ut,u)==e&&(wt=n.Xl(t.L,n.Yl(t.L,ut,u),n.Yl(t.L,1-ut,u)),wt<uA?(M[0]-=wt+m0,M[3]+=wt+n1):wt<0?(M[2]+=wt+m0,M[3]-=wt):wt<m0?(M[1]+=wt,M[2]+=m0-wt):(M[0]+=wt-m0,M[1]+=n1-wt));if(n.Zo(t.L,e)==0?n.Gm(t.L,e)?M[3]-=.2:M[1]-=.2:M[1]-=.1,(bt!=null||Lt!=null)&&(M[1]+=10),(I!=null||s!=null)&&(M[3]+=10),C="",O!=0&&(S=t.ub("H"),v=0,m=t.vb(),O==-1?(C="n",t.Bb((t.X*2+1)/3|0),v=t.ub(C)):O>1&&(C=""+O,t.Bb((t.X*2+1)/3|0),v=t.ub(C)),M[1]<.6||M[3]<.6?(L=n.Ai(t.R,n.Ql(t.L,e)),M[1]<=M[3]?(M[1]+=10,Q=n.zi(t.R,n.Pl(t.L,e))+(Qt+S)/2):(M[3]+=10,Q=n.zi(t.R,n.Pl(t.L,e))-(Qt+S)/2-v)):(Q=n.zi(t.R,n.Pl(t.L,e)),M[0]<M[2]?(M[0]+=10,L=n.Ai(t.R,n.Ql(t.L,e))-m):(M[2]+=10,L=n.Ai(t.R,n.Ql(t.L,e))+m)),v>0&&(xt=Q+(S+v)/2,Nt=L+((t.vb()*4+4)/8|0),n.Id(t,xt,Nt,C,true),t.Bb(t.X)),n.Id(t,Q,L,"H",true)),D=0,vt!=0){for(ot=50,h=0,K=0;K<4;K++)b=K>1?K-2:K+2,M[K]<ot?(D=K,ot=M[K],h=M[b]):M[K]==ot&&M[b]>h&&(D=K,h=M[b]);switch(D){case 0:Q=n.zi(t.R,n.Pl(t.L,e)),L=n.Ai(t.R,n.Ql(t.L,e))-t.V-Qt/2;break;case 1:Q=n.zi(t.R,n.Pl(t.L,e))+t.V+Qt/2,L=n.Ai(t.R,n.Ql(t.L,e));break;case 2:Q=n.zi(t.R,n.Pl(t.L,e)),L=n.Ai(t.R,n.Ql(t.L,e))+t.V+Qt/2;break;default:Q=n.zi(t.R,n.Pl(t.L,e))-t.V-Qt/2,L=n.Ai(t.R,n.Ql(t.L,e));}if(vt==1)U.rj(t.$,new Tt.FN(Q-t.V,L-t.V,2*t.V,2*t.V)),t.K||U.rj(t.U,new n.ie(Q,L,n.wd(t,e)?-3:t.p[e]));else {switch(D){case 2:case 0:Ot=2*t.V,pt=0,Q-=t.V;break;case 1:Ot=0,pt=2*t.V,L-=t.V;break;default:Ot=0,pt=2*t.V,L-=t.V;}U.rj(t.$,new Tt.FN(Q-t.V,L-t.V,2*t.V,2*t.V)),t.K||U.rj(t.U,new n.ie(Q,L,n.wd(t,e)?-3:t.p[e])),U.rj(t.$,new Tt.FN(Q+Ot-t.V,L+pt-t.V,2*t.V,2*t.V)),t.K||U.rj(t.U,new n.ie(Q+Ot,L+pt,n.wd(t,e)?-3:t.p[e]));}}t.C==-8&&n.Vd(t,-9);},n.Fd=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;if(v=new n.je(),A=new n.je(),D=new n.je(),d=new Tt.qN(),g=new Tt.qN(),r=n.Yl(t.L,0,e),i=n.Yl(t.L,1,e),t.yb(e,n.zi(t.R,n.Pl(t.L,r)),n.Ai(t.R,n.Ql(t.L,r)),n.zi(t.R,n.Pl(t.L,i)),n.Ai(t.R,n.Ql(t.L,i))),!n.Om(t.L,r)&&!n.Om(t.L,i)&&kt(G(Pt(n.Nl(t.L,r),n.Nl(t.L,i)),r0),0)&&n.Vd(t,-8),t.o[r]?(v.a=t.o[r].a,v.c=t.o[r].b):(v.a=n.zi(t.R,n.Pl(t.L,r)),v.c=n.Ai(t.R,n.Ql(t.L,r))),t.o[i]?(v.b=t.o[i].a,v.d=t.o[i].b):(v.b=n.zi(t.R,n.Pl(t.L,i)),v.d=n.Ai(t.R,n.Ql(t.L,i))),(n.gm(t.L,e)&Le)!=0){n.Rd(t,v)&&t.qb(v),n.Vd(t,-9);return;}switch(u=n.hm(t.L,e)==8?0:n.hm(t.L,e)==16?1:n.em(t.L,e),u){case 1:if(o=n.hm(t.L,e),(t.G&128)!=0&&(o==257||o==129)&&(C=n.Yl(t.L,0,e),b=n.Gl(t.L,C),b!=0)){for(h=n.Fl(t.L,C),s=0,P=0;P<t.L.f;P++)n.Gl(t.L,P)==b&&n.Fl(t.L,P)==h&&++s;s==1&&(o=1);}switch(o){case 1:n.Rd(t,v)&&n.kd(t,v,r,i);break;case 257:n.Od(t,v,r,i);break;case 129:for(M=v.b-v.a,S=v.d-v.c,n.Cm(t.L,n.Wo(t.L,r,i))?(Q=-3,L=-3):(Q=t.p[r],L=n.qd(t,r),Q==n.Bl(t.L,r)&&(Q=L)),c=2;c<17;c+=2)A.a=v.a+c*M/17-c*S/128,A.c=v.c+c*S/17+c*M/128,A.b=v.a+c*M/17+c*S/128,A.d=v.c+c*S/17-c*M/128,n.Rd(t,A)&&(n.Vd(t,c<9?Q:L),t.pb(A),n.Vd(t,t.Q));break;case 16:n.Rd(t,v)&&n.md(t,v,r,i);}break;case 0:case 2:if((t.s[r]||n.Po(t.L,r)==2)&&(t.s[i]||n.Po(t.L,i)==2)&&!n.fq(t.L,e)&&u==2){if(!n.Rd(t,v))break;n.Ad(t,v.b-v.a,v.d-v.c,d),M=d.a/2,S=d.b/2,A.a=v.a+M,A.c=v.c+S,A.b=v.b+M,A.d=v.d+S,D.a=v.a-M,D.c=v.c-S,D.b=v.b-M,D.d=v.d-S,n.hm(t.L,e)==386&&n.Qd(A,D),n.kd(t,A,r,i),u==2?n.kd(t,D,r,i):n.hd(t,D,r,i);}else (t.s[i]||n.Po(t.L,i)==2)&&u==2?n.Bd(t,v,e,false):(t.s[r]||n.Po(t.L,r)==2)&&u==2?n.Bd(t,v,e,true):(m=n.Cp(t.L,e),m==0&&(m=1),A.a=v.a,A.c=v.c,A.b=v.b,A.d=v.d,n.Ad(t,v.b-v.a,v.d-v.c,d),m>0?(D.a=v.a+d.a,D.c=v.c+d.b,D.b=v.b+d.a,D.d=v.d+d.b,(n.zd(t,r,i,1,g)||n.Zo(t.L,r)>1)&&(D.a+=g.a+d.b,D.c+=g.b-d.a),(n.zd(t,i,r,-1,g)||n.Zo(t.L,i)>1)&&(D.b+=g.a-d.b,D.d+=g.b+d.a)):(D.a=v.a-d.a,D.c=v.c-d.b,D.b=v.b-d.a,D.d=v.d-d.b,(n.zd(t,r,i,-1,g)||n.Zo(t.L,r)>1)&&(D.a+=g.a+d.b,D.c+=g.b-d.a),(n.zd(t,i,r,1,g)||n.Zo(t.L,i)>1)&&(D.b+=g.a-d.b,D.d+=g.b+d.a)),n.hm(t.L,e)==386&&n.Qd(A,D),n.Rd(t,A)&&n.kd(t,A,r,i),u==2?n.Rd(t,D)&&n.kd(t,D,r,i):n.Rd(t,D)&&n.hd(t,D,r,i));break;case 3:n.Rd(t,v)&&(n.kd(t,v,r,i),n.Ad(t,v.b-v.a,v.d-v.c,d),n.ld(t,v,r,i,d.a,d.b,A),n.ld(t,v,r,i,-d.a,-d.b,A));break;case 4:n.Rd(t,v)&&(n.Ad(t,v.b-v.a,v.d-v.c,d),n.ld(t,v,r,i,1.5*d.a,1.5*d.b,A),n.ld(t,v,r,i,.5*d.a,.5*d.b,A),n.ld(t,v,r,i,-0.5*d.a,-0.5*d.b,A),n.ld(t,v,r,i,-1.5*d.a,-1.5*d.b,A));break;case 5:n.Rd(t,v)&&(n.kd(t,v,r,i),n.Ad(t,v.b-v.a,v.d-v.c,d),n.ld(t,v,r,i,2*d.a,2*d.b,A),n.ld(t,v,r,i,d.a,d.b,A),n.ld(t,v,r,i,-d.a,-d.b,A),n.ld(t,v,r,i,-2*d.a,-2*d.b,A));}t.C==-8&&n.Vd(t,-9);},n.Gd=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g;for(b=false,P=0;P<t.L.g;P++)Q=null,n.Bm(t.L,P)?(s=n.$l(t.L,P),L=n.Zl(t.L,P),Q=s==L?"["+s+"]":"["+s+":"+L+"]"):(n.gm(t.L,P)&oA)!=0?Q=(n.gm(t.L,P)&oA)==F0?"a":(n.gm(t.L,P)&384)==256?"r!a":"!a":(n.gm(t.L,P)&384)!=0&&(Q=(n.gm(t.L,P)&384)==256?"r":"!r"),h=(n.gm(t.L,P)&ie)>>17,h!=0&&(Q=(Q??"")+h),Q!=null&&(e=n.Yl(t.L,0,P),A=n.Yl(t.L,1,P),b||(t.Bb((t.X*2+1)/3|0),b=true),c=(n.zi(t.R,n.Pl(t.L,e))+n.zi(t.R,n.Pl(t.L,A)))/2,g=(n.Ai(t.R,n.Ql(t.L,e))+n.Ai(t.R,n.Ql(t.L,A)))/2,i=n.zi(t.R,n.Pl(t.L,A))-n.zi(t.R,n.Pl(t.L,e)),D=n.Ai(t.R,n.Ql(t.L,A))-n.Ai(t.R,n.Ql(t.L,e)),r=j.Math.sqrt(i*i+D*D),o=.6*t.ub(Q),u=.55*t.vb(),r!=0&&(i>0?n.Id(t,c+o*D/r,g-u*i/r,Q,true):n.Id(t,c-o*D/r,g+u*i/r,Q,true)));b&&t.Bb(t.X);},n.Hd=function(t,e,A,P){U.rj(t.$,new Tt.FN(e-t.V,A-t.V,2*t.V,2*t.V)),t.K||U.rj(t.U,new n.ie(e,A,n.wd(t,P)?-3:t.p[P]));},n.Id=function(t,e,A,P,r){var i,D,u;r&&(i=t.ub(P),D=i/2+(t.vb()/8|0),u=t.vb()/2|0,(P=="+"||P=="-")&&(u=u*2/3),U.rj(t.$,new Tt.FN(e-D,A-u,2*D,2*u))),t.K||t.sb(P,e,A);},n.Jd=function(t){var e;e=t.a,t.a=t.b,t.b=e,e=t.c,t.c=t.d,t.d=e;},n.Kd=function(t,e,A){var P;return P=e==0?X1+t[0]-t[t.length-1]:t[e]-t[e-1],A>rr&&A<b0?P-=2*j.Math.cos(A+Hn):P-=.5*j.Math.cos(A+Hn),P;},n.Ld=function(t){var e;return e=new Tt.EN(),t.a<=t.b?(e.c=t.a,e.b=t.b-t.a):(e.c=t.b,e.b=t.a-t.b),t.c<=t.d?(e.d=t.c,e.a=t.d-t.c):(e.d=t.d,e.a=t.c-t.d),e;},n.Md=function(t,e){var A,P,r,i,D,u,o;for(A=z(l.b7,jt,6,n.Ko(t.L,e),15,1),r=0;r<n.Ko(t.L,e);r++)A[r]=n.Xl(t.L,e,n.Yo(t.L,e,r));for(w.qzb(A,w.Xzb(h0(U.arb.prototype.jd,U.arb,[]))),i=n.Nd(A,0),D=n.Kd(A,0,i),P=1;P<A.length;P++)u=n.Nd(A,P),o=n.Kd(A,P,u),D<o&&(D=o,i=u);return i;},n.Nd=function(t,e){var A;return e>0?(t[e]+t[e-1])/2:(A=n1+(t[0]+t[t.length-1])/2,A>n1?A-X1:A);},n.Od=function(t,e,A,P){var r,i,D,u;if(u=new n.je(),!(e.a==e.b&&e.c==e.d)){for(u.a=e.a,u.c=e.c,u.b=e.b,u.d=e.d,D=n.Ld(u),r=0;r<t.$.a.length;r++)if(i=U.wj(t.$,r),!(i.c>D.c+D.b||i.d>D.d+D.a||D.c>i.c+i.b||D.d>i.d+i.a)){if(n.Pd(t,u.a,u.c,r)){if(n.Pd(t,u.b,u.d,r))return;n.Sd(t,u,0,r),n.Od(t,u,A,P);return;}if(n.Pd(t,u.b,u.d,r)){n.Sd(t,u,1,r),n.Od(t,u,A,P);return;}}n.nd(t,u,A,P);}},n.Pd=function(t,e,A,P){var r;return (t.G&1)!=0?false:(r=U.wj(t.$,P),e>r.c&&e<r.c+r.b&&A>r.d&&A<r.d+r.a);},n.Qd=function(t,e){var A;A=t.b,t.b=e.b,e.b=A,A=t.d,t.d=e.d,e.d=A;},n.Rd=function(t,e){var A,P,r,i,D,u;if(e.a==e.b&&e.c==e.d){for(D=new U.wqb(t.$);D.a<D.c.a.length;)if(i=U.vqb(D),Tt.AN(i,e.a,e.c))return  false;return  true;}for(u=n.Ld(e),A=false,e.a>e.b&&(n.Jd(e),A=true),P=0;P<t.$.a.length;P++)if(i=U.wj(t.$,P),!(i.c>u.c+u.b||i.d>u.d+u.a||u.c>i.c+i.b||u.d>i.d+i.a)){if(n.Pd(t,e.a,e.c,P))return n.Pd(t,e.b,e.d,P)?(A&&n.Jd(e),false):(n.Sd(t,e,0,P),r=n.Rd(t,e),A&&n.Jd(e),r);if(n.Pd(t,e.b,e.d,P))return n.Sd(t,e,1,P),r=n.Rd(t,e),A&&n.Jd(e),r;}return A&&n.Jd(e),true;},n.Sd=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c;A==0?(s=e.a,b=e.c,h=e.b,c=e.d):(s=e.b,b=e.d,h=e.a,c=e.c),L=U.wj(t.$,P),o=h>s?L.c+L.b:L.c,Q=c>b?L.d+L.a:L.d,r=h-s,i=c-b,j.Math.abs(r)>j.Math.abs(i)?b==c?(D=o,u=b):(D=s+r*(Q-b)/i,h>s==o>D?u=Q:(D=o,u=b+i*(o-s)/r)):s==h?(D=s,u=Q):(u=b+i*(o-s)/r,c>b==Q>u?D=o:(D=s+r*(Q-b)/i,u=Q)),A==0?(e.a=D,e.c=u):(e.b=D,e.d=u);},n.Td=function(t,e){var A,P,r,i,D,u;if(t.L.q!=0){for(n.aw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),t.B=e,n.ed(t),r=n.fw(t.L),i=false,t.p=z(l.d7,H,6,t.L.q,15,1),A=0;A<t.L.q;A++)t.p[A]=n.Bl(t.L,A),t.p[A]!=0&&(i=true),n.Om(t.L,A)&&(t.p[A]=128),n.qm(t.L,A)&&(t.G&z1)==0&&(t.p[A]=256);for(n.Vd(t,-10),n.td(t),n.sd(t),n.ud(t),n.bd(t),t.Bb(t.X),t.zb(t.Y),n.Vd(t,t.Q),n.xd(t),w.pzb(t.U.a,0),w.pzb(t.$.a,0),(t.G&1)!=0&&(n.Cd(t,r),n.Dd(t),n.Gd(t)),D=0;D<t.L.q;D++)n.wd(t,D)?(n.Vd(t,-3),n.Ed(t,D,r),n.Vd(t,t.Q)):t.p[D]!=0?(n.Vd(t,t.p[D]),n.Ed(t,D,r),n.Vd(t,t.Q)):!i&&t.L.O!=1&&n.Sl(t.L,D)!=1&&n.Sl(t.L,D)!=6&&(t.G&i0)==0&&n.Il(t.L,D)==null&&n.Sl(t.L,D)<n.Xc.length&&n.Xc[n.Sl(t.L,D)]!=0?(u=n.Xc[n.Sl(t.L,D)],n.Yd(t,(P=t.M!=0?t.M:t.D!=0?t.D:-1,lt.N3(u,P))),n.Ed(t,D,r),n.Vd(t,t.Q)):n.Ed(t,D,r);(t.G&1)==0&&(n.Dd(t),n.Gd(t),n.Cd(t,r));}},n.Ud=function(t,e,A,P){var r;switch(r=A/2,P&U1){case U1:if(e){t.w.a=e.c+e.b/2,t.w.b=e.d+e.a-r;break;}case 0:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d+t.v.a+r,e&&t.w.b>e.d+e.a-r&&(t.w.b=e.d+e.a-r);break;case ye:if(e){t.w.a=e.c+e.b/2,t.w.b=e.d+r;break;}case Ue:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d-r,e&&t.w.b<e.d+r&&(t.w.b=e.d+r);}},n.Vd=function(t,e){if(!t.K&&(e==-10&&(t.C=-999,e=t.Q),e!=-2&&e!=-7&&t.N!=0&&(e=-4),e!=t.C&&!(t.C==-8&&e!=-9)))switch(e==-8&&(t.O=t.C),e==-9&&(e=t.O),t.C=e,e){case 0:t.Ab(t.F==0?O0:t.F);break;case  -6:t.Ab(t.F);break;case  -4:t.Ab(t.N);break;case  -2:t.Ab(t.t);break;case  -3:t.Ab(t.u);break;case  -7:t.Ab(t.H);break;case  -8:t.Ab(t.I);break;case  -5:t.Ab(t.P);break;case 64:t.Ab(-14655233);break;case 128:t.Ab(fe);break;case 256:t.Ab(-4194049);break;case 192:t.Ab(-16711936);break;case 320:t.Ab(-24576);break;case 384:t.Ab(-16744448);break;case 448:t.Ab(-6291456);break;case 1:t.Ab(EA);break;default:t.Ab(O0);}},n.Wd=function(t,e){t.G=e;},n.Xd=function(t,e,A){t.Q=-6,t.F=e,t.D=A,n._d(t);},n.Yd=function(t,e){if(t.N!=0){t.C!=-4&&(t.C=-4,t.Ab(t.N));return;}t.C=-5,t.P=e,t.Ab(e);},n.Zd=function(t){var e,A,P,r,i;for(r=n.zi(t.R,n.Pl(t.L,0)),A=n.zi(t.R,n.Pl(t.L,0)),i=n.Ai(t.R,n.Ql(t.L,0)),P=n.Ai(t.R,n.Ql(t.L,0)),e=0;e<t.L.q;e++)r>n.zi(t.R,n.Pl(t.L,e))&&(r=n.zi(t.R,n.Pl(t.L,e))),A<n.zi(t.R,n.Pl(t.L,e))&&(A=n.zi(t.R,n.Pl(t.L,e))),i>n.Ai(t.R,n.Ql(t.L,e))&&(i=n.Ai(t.R,n.Ql(t.L,e))),P<n.Ai(t.R,n.Ql(t.L,e))&&(P=n.Ai(t.R,n.Ql(t.L,e)));return new Tt.FN(r,i,A-r,P-i);},n.$d=function(t,e,A){var P,r;return t.L.q==0?null:(t.v=n.Zd(t),P=t.R.c*n.Tl(t.L),r=new n.Di(t.v,e,P,A),r.c==1&&r.a==0&&r.b==0?r=null:(n.ti(r,t.R),n.wi(r,t.v)),n.Ud(t,e,P,A),r);},n._d=function(t){var e;e=t.M!=0?t.M:t.D!=0?t.D:-1,t.t=lt.O3(e),t.u=lt.N3(KA,e),t.H=-24321,t.I=-6291392;},n.ae=function(t,e,A,P){var r;return n.be(t,e,A,P),n.yi(t.R)?null:(r=t.R,n.ui(t.R,t.L),t.R=new n.Bi(),r);},n.be=function(t,e,A,P){var r,i,D,u;if(t.L.q==0)return null;for(D=t.Cb(A,P),n.aw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),n.xd(t),w.pzb(t.U.a,0),w.pzb(t.$.a,0),t.B=e,n.ed(t),t.Bb(t.X),t.K=true,i=0;i<t.L.q;i++)n.Ed(t,i,null);return t.K=false,r=t.R.c*n.Tl(t.L),n.od(t,r),n.Ud(t,A,r,P),!A||Tt.BN(A,t.v)?D:(u=new n.Di(t.v,A,r,P),n.ti(u,t.R),n.wi(u,t.v),n.vi(u,t.w),D?(n.ti(u,D),D):u);},n.ce=function(t){n.de.call(this,t,0);},n.de=function(t,e){n.ad(this),this.L=t,this.G=e,this.wb();},x(122,1,{122:1}),y.wb=function(){n.vd(this);},y.xb=function(t,e,A,P){},y.yb=function(t,e,A,P,r){},y.Cb=function(t,e){return n.$d(this,t,e);},y.r=0,y.t=0,y.u=0,y.A=0,y.C=0,y.D=0,y.F=0,y.G=0,y.H=0,y.I=0,y.J=0,y.K=false,y.M=0,y.N=0,y.O=0,y.P=0,y.Q=0,y.S=0,y.T=0,y.V=0,y.W=0,y.X=0,y.Y=0,y.Z=0,n.Yc=.15,n.Zc=.06,n.$c=.6,l.i7=X(122),n.ie=function(t,e,A){this.b=t,this.c=e,this.a=A;},x(143,1,{143:1},n.ie),y.a=0,y.b=0,y.c=0,l.g7=X(143),n.je=function(){},x(67,1,{},n.je),y.a=0,y.b=0,y.c=0,y.d=0,l.h7=X(67),n.ke=function(t,e){var A;if(t.i==null)return  false;for(A=0;A<t.i.length;A++)if(!e.Gb(t.i[A].a,t.i[A].b))return  false;return  true;},n.le=function(t,e,A){var P;if(t.i!=null)for(P=0;P<t.i.length;P++)t.i[P].a+=e,t.i[P].b+=A;},n.me=function(t,e){var A;if(t.i!=null)for(A=0;A<t.i.length;A++)t.i[A].a*=e,t.i[A].b*=e;},n.ne=function(t,e,A){var P;if(t.i!=null)for(P=0;P<t.i.length;P++)t.i[P].a=t.o[P]+e-t.k,t.i[P].b=t.p[P]+A-t.n;},n.oe=function(t,e,A){var P,r;if(t.k=e,t.n=A,t.i!=null)for(r=t.i.length,t.o=z(l.b7,jt,6,r,15,1),t.p=z(l.b7,jt,6,r,15,1),P=0;P<r;P++)t.o[P]=t.i[P].a,t.p[P]=t.i[P].b;},n.pe=function(t,e,A){var P,r,i;if(t.i!=null)for(P=0;P<t.i.length;P++)i=t.p[P]*e,r=t.o[P]-A,t.i[P].a=t.k+i*j.Math.sin(r),t.i[P].b=t.n+i*j.Math.cos(r);},n.qe=function(t,e,A){var P,r,i,D;if(t.k=e,t.n=A,t.i!=null)for(D=t.i.length,t.o=z(l.b7,jt,6,D,15,1),t.p=z(l.b7,jt,6,D,15,1),i=0;i<D;i++)P=e-t.i[i].a,r=A-t.i[i].b,t.p[i]=j.Math.sqrt(P*P+r*r),t.o[i]=n.mo(e,A,t.i[i].a,t.i[i].b);},x(193,1,{193:1}),y.Ib=function(t){var e,A;if(this.i==null)return null;for(e=new Tt.FN(this.i[0].a,this.i[0].b,0,0),A=1;A<this.i.length;A++)e.c>this.i[A].a?(e.b+=e.c-this.i[A].a,e.c=this.i[A].a):e.b<this.i[A].a-e.c&&(e.b=this.i[A].a-e.c),e.d>this.i[A].b?(e.a+=e.d-this.i[A].b,e.d=this.i[A].b):e.a<this.i[A].b-e.d&&(e.a=this.i[A].b-e.d);return e;},y.Mb=function(t){n.me(this,t);},y.ob=function(){var t;return t=new T.Gjb(),T.Fjb(t,ir+this.Kb()+'"'+this.Jb()+YA),t.a;},y.Nb=function(t,e){n.ne(this,t,e);},y.Ob=function(t,e){n.pe(this,t,e);},y.Pb=function(t,e){n.qe(this,t,e);},y.g=false,y.j=false,y.k=0,y.n=0,l.j7=X(193),n.xe=function(t,e,A){if(t.f[e])return 0;if(n.Sl(t.j,e)==7){if(n.Ko(t.j,e)==3)return 6;if(n.Zo(t.j,e)==2)return t.a?0:4;}else {if(n.Sl(t.j,e)==8)return 10;if(n.Sl(t.j,e)==15||n.Sl(t.j,e)==33){if(n.Zo(t.j,e)==3)return 8;}else if(n.Sl(t.j,e)==16||n.Sl(t.j,e)==34||n.Sl(t.j,e)==52){if(n.Zo(t.j,e)==2)return 11;if(n.Zo(t.j,e)==3)return n.Al(t.j,e)==1?12:(A&&n.hn(t.j,e,1),5);}else if(n.Sl(t.j,e)==6)return n.Al(t.j,e)==-1?n.Ko(t.j,e)==3?16:n.Lo(t.j,e)==3?15:14:(A&&n.hn(t.j,e,-1),n.Ko(t.j,e)!=n.Lo(t.j,e)?2:3);}return 0;},n.ye=function(t,e,A){if(t.f[e])return 0;if(t.a){if(n.Ko(t.j,e)!=3)return 0;}else if(n.Ko(t.j,e)>3)return 0;return n.Sl(t.j,e)==6?(A&&n.hn(t.j,e,1),2):n.Sl(t.j,e)==5?4:0;},n.ze=function(t,e,A){if(n.Al(t.j,e)!=0)return 0;if(t.a){if(n.Sl(t.j,e)==5)return n.up(t.j,e)!=2?0:(A&&n.hn(t.j,e,1),1);if(n.Sl(t.j,e)==7)return n.up(t.j,e)!=2?0:(A&&n.hn(t.j,e,-1),n.Ce(t,e)?6:3);if(n.Sl(t.j,e)==8)return n.up(t.j,e)!=1?0:(A&&n.hn(t.j,e,-1),n.Ce(t,e)?7:4);if(n.Sl(t.j,e)==16)return n.up(t.j,e)!=1?0:(A&&n.hn(t.j,e,-1),n.Ce(t,e)?5:2);if(n.Sl(t.j,e)==34)return n.up(t.j,e)!=1?0:(A&&n.hn(t.j,e,-1),n.Ce(t,e)?4:1);}else {if(n.Sl(t.j,e)==5)return n.up(t.j,e)>2?0:(A&&n.hn(t.j,e,1),1);if(n.Sl(t.j,e)==7)return n.up(t.j,e)>2?0:(A&&n.hn(t.j,e,-1),n.Ce(t,e)?5:3);if(n.Sl(t.j,e)==8)return n.up(t.j,e)>1?0:(A&&n.hn(t.j,e,-1),n.Ce(t,e)?7:4);if(n.Sl(t.j,e)==16)return n.up(t.j,e)>1?0:(A&&n.hn(t.j,e,-1),n.Ce(t,e)?5:2);}return 0;},n.Ae=function(t,e,A){var P,r,i;if(P=n.Sl(t.j,e),P>=5&&P<=8||P==15||P==16||P==33||P==34||P==52){if(i=n.op(t.j,e),i!=0)return  true;if(r=n.Al(t.j,e),P==5&&r>=0)return A&&n.hn(t.j,e,r-1),true;if(P!=5&&r<=0)return A&&n.hn(t.j,e,r+1),true;}return  false;},n.Be=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c;for(e=0;e<t.j.f;e++)if(t.d[e])for(D=false,u=z(l.d7,H,6,t.j.f,15,1),c=z(l.d7,H,6,t.j.f,15,1),o=z(l.d7,H,6,t.j.f,15,1),u[0]=e,c[e]=-1,o[e]=1,r=0,Q=0;r<=Q&&!D;){for(i=u[r],L=0;L<n.Zo(t.j,i)&&!D;L++)if(s=(o[i]&1)==1^n.em(t.j,n.$o(t.j,i,L))>1,P=n.Yo(t.j,i,L),o[P]==0&&s)if(t.d[P]){if((o[i]&1)==1){for(t.d[e]=false,t.d[P]=false,t.k+=2,b=i,h=0;h<o[i];h++)A=n.Wo(t.j,P,b),t.e[A]&&(t.e[A]=false,--t.c),n.Hn(t.j,A,n.em(t.j,A)==1?2:n.em(t.j,A)-1),P=b,b=c[P];D=true;}}else u[++Q]=P,c[P]=i,o[P]=o[i]+1;++r;}},n.Ce=function(t,e){var A;for(A=0;A<n.Zo(t.j,e);A++)if(n.Km(t.j,n.Yo(t.j,e,A)))return  true;return  false;},n.De=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;if(e!=null)t.e=e;else for(t.e=z(l.zeb,ht,6,t.j.g,16,1),h=0;h<t.j.g;h++)n.hm(t.j,h)==8&&(t.e[h]=true,n.Ln(t.j,h,1));for(b=0;b<t.j.g;b++)t.e[b]&&++t.c;for(t.d=z(l.zeb,ht,6,t.j.f,16,1),t.f=z(l.zeb,ht,6,t.j.f,16,1),u=0;u<t.j.f;u++){for(c=0,g=0;g<n.Zo(t.j,u);g++)t.e[n.$o(t.j,u,g)]&&++c;c>0&&(t.d[u]=true,++t.b,c==3&&(t.f[u]=true));}for(t.i=z(l.zeb,ht,6,A.j.a.length,16,1),m=0;m<A.j.a.length;m++)for(t.i[m]=true,Q=U.wj(A.k,m),L=0,s=Q.length;L<s;++L)if(o=Q[L],!t.e[o]){t.i[m]=false;break;}for(t.g=z(l.zeb,ht,6,t.j.f,16,1),d=0;d<A.j.a.length;d++)if(t.i[d]&&U.wj(A.k,d).length!=6)for(r=U.wj(A.j,d),i=0,D=r.length;i<D;++i)P=r[i],U.wj(A.k,d).length==5&&(t.g[P]=true);t.k=0;},n.Ee=function(t,e,A){var P,r;for(r=0;r<n.Zo(t.j,e);r++)if(P=n.Yo(t.j,e,r),n.Sl(t.j,P)==8&&n.Zo(t.j,P)==1)return A&&n.Al(t.j,P)==0&&n.hn(t.j,P,-1),true;return  false;},n.Fe=function(t,e,A){var P,r,i,D;if(n.aw(t.j,1),D=new n.Ct(t.j,1),n.De(t,e,D),t.c==0)return  true;if(A)for(r=0;r<t.j.f;r++)t.d[r]&&n.Sl(t.j,r)==7&&(t.f[r]||n.Zo(t.j,r)==3&&!t.g[r])&&n.hn(t.j,r,1);for(t.a=false,n.Qe(t,A),t.j.K&&n.Ie(t),n.Se(t,D),n.Ne(t,D),n.Pe(t),n.Je(t);n.Le(t,D,A);)n.Je(t);for(;n.Me(t,D,A);)n.Je(t);for(;n.Ke(t,D,A);)n.Je(t);for(;t.c!=0;)for(i=0;i<t.j.g;i++)t.e[i]&&(n.He(t,i),n.Je(t));for(A&&n.Te(t,A),t.b-t.k>=2&&n.Be(t),P=0;P<t.j.f;P++)t.d[P]&&n.np(t.j,P)!=0&&(A&&(n.Al(t.j,P)==1&&n.Gm(t.j,P)||n.Al(t.j,P)==-1&&n.Sl(t.j,P)==5)?n.hn(t.j,P,0):n.vn(t.j,P,32),++t.k);return t.b==t.k;},n.Ge=function(t,e){return t==e-1?0:t+1;},n.He=function(t,e){var A,P,r,i;for(n.hm(t.j,e)==1&&(n.Ln(t.j,e,2),t.k+=2),r=0;r<2;r++)for(A=n.Yl(t.j,r,e),t.d[A]=false,i=0;i<n.Zo(t.j,A);i++)P=n.$o(t.j,A,i),t.e[P]&&(t.e[P]=false,--t.c);},n.Ie=function(t){var e,A,P,r,i,D,u,o;for(A=0;A<t.j.g;A++)if(t.e[A])for(r=0;r<2;r++){for(u=n.Yl(t.j,r,A),e=false,D=0;D<n.Zo(t.j,u);D++)if(A!=n.$o(t.j,u,D)&&t.e[n.$o(t.j,u,D)]){e=true;break;}if(!e){for(o=A,P=n.Yl(t.j,1-r,A);o!=-1;)for(t.d[u]=false,t.e[o]=false,--t.c,n.Ln(t.j,o,8),o=-1,u=P,i=0;i<n.Zo(t.j,u);i++)if(t.e[n.$o(t.j,u,i)])if(o==-1)o=n.$o(t.j,u,i),P=n.Yo(t.j,u,i);else {u=-1,o=-1;break;}u!=-1&&(t.d[P]=false);break;}}},n.Je=function(t){var e,A,P,r,i,D,u;do for(u=false,A=0;A<t.j.g;A++)if(t.e[A]){for(i=false,r=0;r<2;r++){for(P=n.Yl(t.j,r,A),e=false,D=0;D<n.Zo(t.j,P);D++)if(A!=n.$o(t.j,P,D)&&t.e[n.$o(t.j,P,D)]){e=true;break;}if(!e){i=true;break;}}i&&(u=true,n.He(t,A));}while(u);},n.Ke=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(c=0;c<e.j.a.length;c++)if(U.wj(e.k,c).length!=6&&t.i[c]){for(s=true,d=U.wj(e.k,c),L=0;L<d.length;L++)if(!t.e[d[L]]){s=false;break;}if(s){for(u=-1,o=0,g=U.wj(e.j,c),Q=0;Q<g.length;Q++)P=g[Q],b=t.g[P]?n.xe(t,P,false):n.ye(t,P,false),o<b&&(o=b,u=Q);if(u!=-1){if(h=g[u],A){for(r=g,i=0,D=r.length;i<D;++i)P=r[i],P!=h&&n.Ae(t,P,true);t.g[h]?n.xe(t,h,true):n.ye(t,h,true);}return n.Oe(t,h),t.g[h]&&n.Re(t,g,d,u),true;}}}return  false;},n.Le=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;for(C=z(l.d7,H,6,t.j.g,15,1),g=0;g<e.j.a.length;g++){for(m=U.wj(e.k,g),s=true,o=0;o<m.length;o++)if(!t.e[m[o]]){s=false;break;}if(s)for(u=0;u<m.length;u++)++C[m[u]];}for(r=t.c,c=0;c<e.j.a.length;c++)for(D=false,d=U.wj(e.j,c),m=U.wj(e.k,c),u=0;u<m.length&&!D;u++)if(C[m[u]]>1&&(i=n.Ge(u,m.length),C[m[i]]==1)){for(Q=false,b=n.Ge(i,m.length);C[m[b]]==1;)(!n.Ae(t,d[b],false)||m.length!=6&&n.Sl(t.j,d[b])!=6)&&(Q=true),b=n.Ge(b,m.length);if(!Q&&(P=b>i?b-i:b+m.length-i,P>2&&(P&1)==1)){for(h=1;h<P;h+=2)L=i+h<m.length?i+h:i+h-m.length,A&&(n.Ae(t,d[L],true),n.Ae(t,d[L==d.length-1?0:L+1],true)),n.He(t,m[L]);D=true;}}return r!=t.c;},n.Me=function(t,e,A){var P,r,i,D,u,o,Q,L;for(P=false,o=0;o<e.j.a.length;o++)if(U.wj(e.k,o).length==6){for(u=true,Q=U.wj(e.j,o),L=U.wj(e.k,o),i=0;i<6;i++)if(!n.Ae(t,Q[i],false)||!t.e[L[i]]){u=false;break;}if(u){if(A)for(D=0;D<6;D++)n.Ae(t,Q[D],true);for(r=0;r<6;r+=2)n.He(t,L[r]);P=true;break;}}return P;},n.Ne=function(t,e){var A;for(A=0;A<t.j.g;A++)t.e[A]&&n.vt(e,A)&&(n.Oe(t,n.Yl(t.j,0,A)),n.Oe(t,n.Yl(t.j,1,A)));},n.Oe=function(t,e){var A,P;for(t.d[e]&&(t.d[e]=false,--t.b),P=0;P<n.Zo(t.j,e);P++)A=n.$o(t.j,e,P),t.e[A]&&(t.e[A]=false,--t.c);},n.Pe=function(t){var e,A,P,r,i;for(A=0;A<t.j.g;A++)if(n.em(t.j,A)==2){for(r=0;r<2;r++)if(e=n.Yl(t.j,r,A),n.Sl(t.j,e)<=8){for(i=0;i<n.Zo(t.j,e);i++)if(P=n.$o(t.j,e,i),t.e[P]){n.Oe(t,e);break;}}}},n.Qe=function(t,e){var A;for(A=0;A<t.j.f;A++)t.d[A]&&n.op(t.j,A)==0&&(!e||n.Sl(t.j,A)==5&&n.Al(t.j,A)<0||n.Sl(t.j,A)==6||n.Sl(t.j,A)==14||n.Gm(t.j,A)&&n.Al(t.j,A)>0)&&n.Oe(t,A);},n.Re=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d;if(s=e[P==0?4:P-1],d=e[P==4?0:P+1],n.Zo(t.j,s)==3&&n.Zo(t.j,d)==3){for(b=z(l.d7,H,6,t.j.f,15,1),h=z(l.zeb,ht,6,t.j.g,16,1),i=A,D=0,u=i.length;D<u;++D)o=i[D],h[o]=true;for(r=0;r<t.j.g;r++)t.e[r]||(h[r]=true);if(c=n.xp(t.j,b,s,d,14,null,h),c==14){for(g=true,L=2;L<=12;L++)if((L&3)!=1^n.To(t.j,b[L])==5){g=false;break;}if(g)for(n.Oe(t,b[7]),Q=1;Q<=5;Q+=2)n.He(t,n.Wo(t.j,b[Q],b[Q+1])),n.He(t,n.Wo(t.j,b[Q+7],b[Q+8]));}}},n.Se=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt;for(v=z(l.zeb,ht,6,t.j.f,16,1),N=0;N<e.j.a.length;N++)if(I=U.wj(e.k,N).length,(I==3||I==5||I==6||I==7)&&t.i[N])for(P=U.wj(e.j,N),r=0,i=P.length;r<i;++r)A=P[r],v[A]=true;for(h=z(l.d7,H,6,t.j.f,15,1),c=z(l.zeb,ht,6,t.j.f,16,1),Q=0;Q<t.j.g;Q++)D=n.Yl(t.j,0,Q),u=n.Yl(t.j,1,Q),!v[D]&&!v[u]&&(t.e[Q]&&(++h[D],++h[u]),n.hm(t.j,Q)==16&&(c[D]=true,c[u]=true));for(K=z(l.d7,H,6,t.j.f,15,1),b=z(l.d7,H,6,t.j.f,15,1),Qt=0;Qt<t.j.f;Qt++)if(h[Qt]==1)for(b[0]=Qt,s=0,g=0;s<=g;){for(d=0;d<n.Zo(t.j,b[s]);d++)if(t.e[n.$o(t.j,b[s],d)]&&(L=n.Yo(t.j,b[s],d),(s==0||L!=b[s-1])&&h[L]!=0)){if(b[++g]=L,(h[L]&1)!=0){for(S=1;S<g;S+=2)K[b[S]]=-1;g=0;}break;}++s;}for(o=z(l.zeb,ht,6,t.j.f,16,1),ut=0;ut<t.j.f;ut++)if(!o[ut]&&h[ut]!=0){for(b[0]=ut,o[ut]=true,s=0,g=0;s<=g;){for(d=0;d<n.Zo(t.j,b[s]);d++)t.e[n.$o(t.j,b[s],d)]&&(L=n.Yo(t.j,b[s],d),o[L]||(b[++g]=L,o[L]=true));++s;}if((g&1)==0){for(m=0;m<=g;m++)K[b[m]]==0&&(K[b[m]]=n.ze(t,b[m],false));for(M=true,C=0;C<=g;C++)if(K[b[C]]<=0&&!n.Ae(t,b[C],false)){M=false;break;}if(M){for(O=0,V=-1,d=0;d<=g;d++)O<K[b[d]]&&(O=K[b[d]],V=b[d]);O>0&&(n.ze(t,V,true),n.Oe(t,V));}}}},n.Te=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g;for(A=0;A<t.j.f;A++)if(n.aw(t.j,1),n.Sl(t.j,A)==7&&n.Al(t.j,A)==1&&n.Po(t.j,A)==1&&!n.Ee(t,A,e))for(u=false,o=z(l.d7,H,6,t.j.f,15,1),g=z(l.d7,H,6,t.j.f,15,1),Q=z(l.d7,H,6,t.j.f,15,1),o[0]=A,g[A]=-1,Q[A]=1,i=0,L=0;i<=L&&!u;){for(D=o[i],s=0;s<n.Zo(t.j,D)&&!u;s++)if(h=(Q[D]&1)==0^n.em(t.j,n.$o(t.j,D,s))>1,r=n.Yo(t.j,D,s),Q[r]==0&&h)if(n.Sl(t.j,r)==7&&n.Po(t.j,r)==0&&n.Zo(t.j,r)==2){if((Q[D]&1)==0){for(n.hn(t.j,A,0),c=D,b=0;b<Q[D];b++)P=n.Wo(t.j,r,c),n.Hn(t.j,P,n.em(t.j,P)==1?2:n.em(t.j,P)-1),r=c,c=g[r];u=true;}}else o[++L]=r,g[r]=D,Q[r]=Q[D]+1;++i;}},n.Ue=function(t){n.Ve.call(this,t);},n.Ve=function(t){this.j=t;},x(89,1,{},n.Ue),y.a=false,y.b=0,y.c=0,y.k=0,l.k7=X(89),n.We=function(t,e){var A,P;for(A=0,P=0;P<t.j[e];P++)t.n[e][P]==2&&(n.Sl(t,t.i[e][P])==7||n.Sl(t,t.i[e][P])==8||n.Sl(t,t.i[e][P])==16)&&++A;return A;},n.Xe=function(t,e){var A,P,r,i,D,u;if(t.s[e]==0)return  false;for(u=true,A=t.s[e],i=t.j[e],D=0,P=0;P<i;P++)r=t.i[e][P],D+=t.s[r];return j.Math.abs(A)<=j.Math.abs(D)&&T.Nib(A)!=T.Nib(D)&&(u=false),u;},n.Ye=function(t,e){var A,P,r,i,D,u,o;if(t.C[e]!=8||t.s[e]!=0||t.j[e]!=1||t.n[e][0]!=1)return  false;if(A=t.i[e][0],t.C[A]==6){for(D=t.j[A],P=0;P<D;P++)if(r=t.i[A][P],r!=e&&t.C[r]==8&&(i=n.Wo(t,A,r),t.H[i]==2))return  true;}else if(t.C[A]==7){if(t.s[A]==1)return  true;}else if(t.C[A]==16){for(u=t.j[A],o=0,P=0;P<u;P++)r=t.i[A][P],r!=e&&t.C[r]==8&&(i=n.Wo(t,A,r),t.H[i]==2&&++o);if(o==2)return  true;}else if(n.Ze(t,e))return  true;return  false;},n.Ze=function(t,e){var A,P,r,i,D;if(t.C[e]!=8||t.j[e]!=1)return  false;if(A=t.i[e][0],t.C[A]==15){for(D=t.j[A],P=0;P<D;P++)if(r=t.i[A][P],r!=e&&t.C[r]==8&&(i=n.Wo(t,A,r),t.H[i]==2))return  true;}return  false;},n.$e=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O;if(t.C[e]!=7||t.s[e]!=0||t.j[e]+t.o[e]>3)return  false;if(e<t.f&&n.nt(t.p,e)){if(t.o[e]!=1||n.So(t,e,7)!=1)return  false;for(M=(n.aw(t,7),t.p),C=0;C<M.j.a.length;C++)if(n.pt(M,C,e)){if(U.wj(M.k,C).length==5||U.wj(M.k,C).length==6){for(S=U.wj(M.j,C),d=-1,o=0;o<S.length;o++)if(S[o]==e){d=o;break;}for(r=0,m=null,g=null,S.length==5&&(m=z(l.d7,H,6,2,15,1),m[0]=S[d-1<0?d+4:d-1],m[1]=S[d-4<0?d+1:d-4],g=z(l.d7,H,6,2,15,1),g[0]=S[d-2<0?d+3:d-2],g[1]=S[d-3<0?d+2:d-3]),S.length==6&&(m=z(l.d7,H,6,3,15,1),m[0]=S[d-1<0?d+5:d-1],m[1]=S[d-3<0?d+3:d-3],m[2]=S[d-5<0?d+1:d-5],g=z(l.d7,H,6,2,15,1),g[0]=S[d-2<0?d+4:d-2],g[1]=S[d-4<0?d+2:d-4]),Q=0;Q<S.length;Q++)e!=S[Q]&&n.Sl(t,S[Q])==7&&n.Po(t,S[Q])==1&&--r;for(L=0;L<m.length;L++){for(i=-1,D=-1,c=0;c<n.Zo(t,m[L]);c++)if(!n.Tp(t,n.$o(t,m[L],c))){i=n.Yo(t,m[L],c),D=n.$o(t,m[L],c);break;}if(i!=-1){if(t.C[i]==7&&t.o[i]==0&&t.j[i]+t.o[i]<=3&&!n._e(t,i,false)){++r;continue;}if(t.C[i]==8&&t.j[i]==1){r+=2;continue;}if(D<t.g&&n.ot(t.p,D)){for(V=0;V<M.j.a.length;V++)if(M.e[V]&&n.pt(M,V,i)){for(v=U.wj(M.j,V),b=0;b<v.length;b++)if(n.Sl(t,v[b])==7&&n.Po(t,v[b])==1){--r;break;}break;}}}}for(s=0;s<g.length;s++){for(i=-1,b=0;b<n.Zo(t,g[s]);b++)n.Tp(t,n.$o(t,g[s],b))||(i=n.Yo(t,g[s],b));n.Sl(t,g[s])==6?i!=-1&&n.We(t,i)!=0&&--r:n.Sl(t,g[s])==7&&n.Po(t,g[s])==0&&(i==-1||!(i<t.f&&n.nt(t.p,i))&&n.We(t,i)==0)&&++r;}return r>0;}break;}return  false;}if(t.o[e]>1)return  false;if(t.o[e]==1){for(h=-1,O=0,o=0;o<t.j[e];o++){if(P=t.i[e][o],t.n[e][o]==2){if(t.C[P]!=6)return  false;h=P;continue;}if(t.C[P]==8)return  false;if(t.C[P]==7){--O,n._e(t,P,false)&&--O;continue;}P<t.f&&n.nt(t.p,P)&&--O;}if(h==-1)return  false;for(A=0,Q=0;Q<t.j[h];Q++)if(t.n[h][Q]==1){if(P=t.i[h][Q],n.We(t,P)!=0)return  false;P<t.f&&n.nt(t.p,P)&&++A,t.C[P]==7&&!n._e(t,P,true)&&++O,(t.C[P]==8||t.C[P]==16)&&--O;}return A==2&&--O,O>=0;}for(u=0;u<t.j[e];u++)if(P=t.i[e][u],P<t.f&&n.nt(t.p,P)||t.C[P]!=6||n.We(t,P)!=0||t.o[P]!=0&&n.af(t,P))return  false;return  true;},n._e=function(t,e,A){var P,r,i;for(P=false,i=0;i<t.j[e];i++)if(!n.Tp(t,t.k[e][i])&&t.n[e][i]==1&&(r=t.i[e][i],!(r<t.f&&n.nt(t.p,r))&&(t.C[r]==6&&n.We(t,r)==1||t.C[r]==16&&n.We(t,r)==2))){if(P||!A)return  true;P=true;}return  false;},n.af=function(t,e){var A,P,r;for(P=0;P<t.j[e];P++)if(t.n[e][P]!=1){for(A=t.i[e][P],r=0;r<t.j[A];r++)if(t.n[A][r]==1&&n.We(t,t.i[A][r])!=0)return  true;}return  false;},n.df=function(){n.df=o1,n.bf=F(k(l.yeb,1),Kt,6,15,[-1,-1,-1,0,0,1,2,3,4,5,-1,0,0,0,6,7,8,9,-1,0,0,10,10,10,10,10,10,10,10,10,10,1,11,11,12,13,-1,0,0,10,10,10,10,10,10,10,10,10,10,0,0,0,11,14,-1,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,10,10,10,10,10,10,10,10,1,1,1,1,-1,-1,-1,-1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]),n.cf=F(k(l.yeb,1),Kt,6,15,[-1,-1,-1,0,0,0,2,5,5,5,-1,0,0,0,0,9,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);},n.ef=function(t,e,A){n.df();var P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(n.aw(t,7),b=z(l.e7,n0,6,t.j[e],14,1),h=0,o=0;o<t.j[e];o++)if(r=t.i[e][o],t.C[r]!=1){if(D=0,(A&32)!=0&&(i=Ct(t.n[e][o]),A==32190?et(i,3)<0&&n.Xp(t,t.k[e][o])&&t.o[e]==1&&(i=0):et(i,3)<0&&n.Tp(t,t.k[e][o])&&(i=0),D=Pt(D,b1(i,4))),(A&128)!=0){if(n.bf[t.C[r]]==-1)throw Ut(new T.bC(fr+t.C[r]));D=B1(D,Ct(n.bf[t.C[r]]));}else if((A&64)!=0){if(n.cf[t.C[r]]==-1)throw Ut(new T.bC(fr+t.C[r]));D=B1(D,Ct(n.cf[t.C[r]]));}for((A&256)!=0&&(c=t.j[r]-1,c>3&&(c=3),(A&512)==0&&c>1&&(c=1),D=Pt(D,l.Oeb(c<<6))),(A&i0)!=0&&(t.u[r]&8)!=0&&(D=Pt(D,256)),(A&z1)!=0&&r<t.f&&n.nt(t.p,r)&&(D=Pt(D,512)),(A&x0)!=0&&(t.u[r]&Q0)!=0&&(D=Pt(D,i0)),L=0;et(D,b[L])<0;)++L;for(s=o;s>L;s--)b[s]=b[s-1];b[L]=D,++h;}for(h>4&&(h=4),P=0,Q=0;Q<h;Q++)P=b1(P,11),P=Pt(P,b[Q]);if(P=b1(P,15),n.bf[t.C[e]]==-1)throw Ut(new T.bC(fr+t.C[e]));if(P=Pt(P,Ct(n.bf[t.C[e]])),(A&2)!=0?(d=t.p&&e<t.f?n.ct(t.p,e):0,d>9&&(d=9),d>2&&(d-=2),P=Pt(P,l.Oeb(d<<4))):(A&1)!=0&&(t.u[e]&8)!=0&&(P=Pt(P,64)),(A&4)!=0&&e<t.f&&n.nt(t.p,e)&&(P=Pt(P,i0)),(A&8)!=0&&(t.u[e]&Y1)!=0&&(P=Pt(P,z1)),(A&16)!=0&&(t.u[e]&Q0)!=0&&(P=Pt(P,Y1)),(A&Y1)!=0&&(n.Xe(t,e)&&(P=Pt(P,Q0)),n.$e(t,e))){for(u=0;u<t.f;u++)if(n.Ye(t,u)){P=Pt(P,x0);break;}}return (A&Q0)!=0&&(g=Ct(n.So(t,e,10)),P=Pt(P,b1(g,7))),P;},n.ff=function(t,e,A,P,r,i){var D,u,o,Q;for(o=0,u=0;u<t.R.f;u++)kt(G(n.Nl(t.R,t.B[u]),P),0)&&++o;if(o!=0)for(n.dg(t,e),n.$f(t,l.Oeb(o),A),D=0;D<t.R.f;D++)Q=G(n.Nl(t.R,t.B[D]),P),et(Q,0)!=0&&(n.$f(t,l.Oeb(D),A),r!=1&&n.$f(t,t0(Q,i),r));},n.gf=function(t,e,A,P,r,i){var D,u,o,Q;for(o=0,u=0;u<t.R.g;u++)(n.gm(t.R,t.C[u])&P)!=0&&++o;if(o!=0)for(n.dg(t,e),n.$f(t,l.Oeb(o),A),D=0;D<t.R.g;D++)Q=n.gm(t.R,t.C[D])&P,Q!=0&&(n.$f(t,l.Oeb(D),A),r!=1&&n.$f(t,l.Oeb(Q>>i),r));},n.hf=function(t){for(var e,A,P,r,i,D;;){for(i=z(l.zeb,ht,6,t.U+1,16,1),P=-1,A=0;A<t.R.f;A++)n.op(t.R,A)!=0&&(i[t.f[A]]&&P<t.f[A]&&(P=t.f[A]),i[t.f[A]]=true);if(P==-1)break;for(r=0,e=0;e<t.R.f;e++)D=0,t.f[e]==P&&(D=++r),n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),n._g(t.e[e],8,l.Oeb(D));t.U=n.Jf(t);}},n.jf=function(t){var e,A,P,r;for(A=0;A<t.R.f;A++)n.dh(t.e[A],A),n._g(t.e[A],2*t.c+4,b1(Ct(t.f[A]),t.c+4));for(r=false,e=0;e<t.R.f;e++)r=r|n.wf(t,e,3);for(P=0;P<t.R.g;P++)r=r|n.rf(t,P,3);if(!r)return  false;for(;t.U<t.R.f&&(r=n.Gf(t),!!r);)n.If(t),t.P&&n.th(t.P,t.f);return  true;},n.kf=function(t){var e,A,P,r,i;for(A=0;A<t.R.f;A++)n.dh(t.e[A],A),n._g(t.e[A],t.c+1,N0(2,Ct(t.f[A])));for(i=z(l.d7,H,6,t.U+1,15,1),P=0;P<t.R.f;P++)++i[t.f[P]];for(r=1;i[r]==1;)++r;for(e=0;e<t.R.f;e++)if(t.f[e]==r){n.ah(t.e[e],1);break;}t.U=n.Jf(t);},n.lf=function(t,e,A){var P,r,i,D,u;return n.Sl(t.R,e)!=6&&n.Sl(t.R,e)!=7||(r=n.Yo(t.R,e,0),i=n.Yo(t.R,e,1),n.Po(t.R,r)!=1||n.Po(t.R,i)!=1)||n.Zo(t.R,r)==1||n.Zo(t.R,i)==1||n.Ko(t.R,r)>3||n.Ko(t.R,i)>3||(D=new n.kk(t.R,t.f,e,r),D.f&&A==1)||(u=new n.kk(t.R,t.f,e,i),u.f&&A==1)||D.f&&u.f?false:(A==3&&(D.f&&D.c&&(t.W[e]=true),u.f&&u.c&&(t.W[e]=true)),P=t.jb?n.nf(t,D,u):n.mf(D,u),A==1?t.db[e]=P:A==2&&(D.f&&(P==1?n.ah(t.e[D.b],Ct(t.f[r])):n.ah(t.e[D.d],Ct(t.f[r]))),u.f&&(P==2?n.ah(t.e[u.b],Ct(t.f[i])):n.ah(t.e[u.d],Ct(t.f[i])))),true);},n.mf=function(t,e){var A,P,r;if(P=n.jk(t),r=n.jk(e),P==-1||r==-1||(P+r&1)==0)return 3;switch(A=0,P+r){case 3:case 7:A=2;break;case 5:A=1;}return A;},n.nf=function(t,e,A){var P,r;return P=z(l.d7,H,6,4,15,1),P[0]=e.b,P[1]=e.a,P[2]=A.a,P[3]=A.b,r=n.dl(t.R,P),j.Math.abs(r)<.3||j.Math.abs(r)>O8?3:r<0?2:1;},n.of=function(t,e,A){var P,r,i,D,u;return !n.Up(t.R,e)||(P=n.Yl(t.R,0,e),r=n.Yl(t.R,1,e),D=new n.kk(t.R,t.f,P,r),D.f&&A==1)||(u=new n.kk(t.R,t.f,r,P),u.f&&A==1)||D.f&&u.f?false:(A==3&&(D.f&&(t.V[e]=n.wg(t,r)),u.f&&(t.V[e]=n.wg(t,P))),i=t.jb?n.qf(t,D,u):n.pf(D,u),A==1?t.o[e]=i:A==2&&(D.f&&(i==2?n.ah(t.e[D.b],Ct(t.f[r])):n.ah(t.e[D.d],Ct(t.f[r]))),u.f&&(i==2?n.ah(t.e[u.b],Ct(t.f[P])):n.ah(t.e[u.d],Ct(t.f[P])))),true);},n.pf=function(t,e){var A,P,r;if(P=n.jk(t),r=n.jk(e),P==-1||r==-1||(P+r&1)==0)return 3;switch(A=0,P+r){case 3:case 7:A=1;break;case 5:A=2;}return A;},n.qf=function(t,e,A){var P,r;return P=z(l.d7,H,6,4,15,1),P[0]=e.b,P[1]=e.a,P[2]=A.a,P[3]=A.b,r=n.dl(t.R,P),j.Math.abs(r)<.3||j.Math.abs(r)>O8?3:r<0?1:2;},n.rf=function(t,e,A){var P,r,i,D,u,o,Q,L;if(t.o[e]!=0)return  false;if(t.R.K){for(Q=0;Q<2;Q++)for(P=n.Yl(t.R,Q,e),L=0;L<n.Ko(t.R,P);L++)if(n.Im(t.R,n.Yo(t.R,P,L)))return  false;}return n.em(t.R,e)==1?n.of(t,e,A):n.em(t.R,e)!=2||n.Tp(t.R,e)||(i=n.Yl(t.R,0,e),D=n.Yl(t.R,1,e),n.Zo(t.R,i)==1||n.Zo(t.R,D)==1)||n.Zo(t.R,i)>3||n.Zo(t.R,D)>3||n.Po(t.R,i)==2||n.Po(t.R,D)==2||(u=new n.kk(t.R,t.f,D,i),u.f&&A==1)||(o=new n.kk(t.R,t.f,i,D),o.f&&A==1)||u.f&&o.f?false:(A==3&&(u.f&&u.c&&(t.V[e]=true),o.f&&o.c&&(t.V[e]=true)),r=n.Fm(t.R,e)?3:t.jb?n.tf(t,u,o):n.sf(u,o),A==1?t.o[e]=r:A==2&&(u.f&&(r==1?n.ah(t.e[u.b],Ct(t.f[i])):r==2&&n.ah(t.e[u.d],Ct(t.f[i]))),o.f&&(r==1?n.ah(t.e[o.b],Ct(t.f[D])):r==2&&n.ah(t.e[o.d],Ct(t.f[D])))),true);},n.sf=function(t,e){return n.jk(t)==-1||n.jk(e)==-1||((n.jk(t)|n.jk(e))&1)!=0?3:n.jk(t)==n.jk(e)?1:2;},n.tf=function(t,e,A){var P,r,i,D,u,o,Q;return i=z(l.b7,jt,6,3,15,1),i[0]=n.Pl(t.R,A.a)-n.Pl(t.R,e.a),i[1]=n.Ql(t.R,A.a)-n.Ql(t.R,e.a),i[2]=n.Rl(t.R,A.a)-n.Rl(t.R,e.a),o=z(l.b7,jt,6,3,15,1),o[0]=n.Pl(t.R,e.b)-n.Pl(t.R,e.a),o[1]=n.Ql(t.R,e.b)-n.Ql(t.R,e.a),o[2]=n.Rl(t.R,e.b)-n.Rl(t.R,e.a),Q=z(l.b7,jt,6,3,15,1),Q[0]=n.Pl(t.R,A.b)-n.Pl(t.R,A.a),Q[1]=n.Ql(t.R,A.b)-n.Ql(t.R,A.a),Q[2]=n.Rl(t.R,A.b)-n.Rl(t.R,A.a),D=z(l.b7,jt,6,3,15,1),D[0]=i[1]*o[2]-i[2]*o[1],D[1]=i[2]*o[0]-i[0]*o[2],D[2]=i[0]*o[1]-i[1]*o[0],u=z(l.b7,jt,6,3,15,1),u[0]=i[1]*D[2]-i[2]*D[1],u[1]=i[2]*D[0]-i[0]*D[2],u[2]=i[0]*D[1]-i[1]*D[0],P=(o[0]*u[0]+o[1]*u[1]+o[2]*u[2])/(j.Math.sqrt(o[0]*o[0]+o[1]*o[1]+o[2]*o[2])*j.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),r=(Q[0]*u[0]+Q[1]*u[1]+Q[2]*u[2])/(j.Math.sqrt(Q[0]*Q[0]+Q[1]*Q[1]+Q[2]*Q[2])*j.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),P<0^r<0?1:2;},n.uf=function(t,e){var A,P,r,i;return A=n.yl(t.R,e),P=n.lp(t.R,e,false),r=n.lp(t.R,e,true),i=-1,P!=r?A!=-1&&A>P?i=A<<24>>24:i=P<<24>>24:A!=-1?(A>r||A<r&&A>=n.up(t.R,e))&&(i=A<<24>>24):!n.zq(t.R,e)&&n.bp(t.R,e)!=0&&(i=n.up(t.R,e),i-=n.lm(t.R,e,i)),n.Qf(t,e,i),i;},n.vf=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(P=z(l.d7,H,6,t.O,15,1),e=0;e<t.R.f;e++){for(L=n.Zo(t.R,e)+n.qp(t.R,e),Q=0,i=0;i<n.Lo(t.R,e);i++)if(i<n.Zo(t.R,e)||i>=n.Ko(t.R,e)){for(s=2*t.f[n.Yo(t.R,e,i)],A=n.$o(t.R,e,i),n.em(t.R,A)==2&&(n.Tp(t.R,A)||++s),u=0;u<Q&&!(s<P[u]);u++);for(o=Q;o>u;o--)P[o]=P[o-1];P[u]=s,++Q;}for(n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),D=L;D<t.O;D++)n._g(t.e[e],t.c+1,0);for(r=0;r<L;r++)n._g(t.e[e],t.c+1,l.Oeb(P[r]));}},n.wf=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g;if(t.db[e]!=0||n.Sl(t.R,e)!=5&&n.Sl(t.R,e)!=6&&n.Sl(t.R,e)!=7&&n.Sl(t.R,e)!=14&&n.Sl(t.R,e)!=15&&n.Sl(t.R,e)!=16||(t.Q&Y1)!=0&&n.Sl(t.R,e)>=14)return  false;if(n.Po(t.R,e)!=0){if(n.Vp(t.R,e))return n.lf(t,e,A);if(n.Sl(t.R,e)!=15&&n.Sl(t.R,e)!=16)return  false;}if(n.Zo(t.R,e)<3||n.Ko(t.R,e)>4||n.Al(t.R,e)>0&&n.Sl(t.R,e)==6||n.Sl(t.R,e)==5&&n.Ko(t.R,e)!=4)return  false;if(t.R.K){if(n.Im(t.R,e))return  false;for(u=0;u<n.Ko(t.R,e);u++)if(n.Im(t.R,n.Yo(t.R,e,u)))return  false;}if(n.Sl(t.R,e)==7&&!t.S[e])return  false;for(c=z(l.d7,H,6,4,15,1),g=z(l.d7,H,6,4,15,1),L=z(l.zeb,ht,6,4,16,1),o=0;o<n.Ko(t.R,e);o++){for(i=-1,r=0,Q=0;Q<n.Ko(t.R,e);Q++)L[Q]||i<t.f[n.Yo(t.R,e,Q)]&&(i=t.f[n.Yo(t.R,e,Q)],r=Q);c[o]=r,g[o]=i,L[r]=true;}if(n.Ko(t.R,e)==4&&g[0]==g[1]&&g[2]==g[3]||n.Ko(t.R,e)==4&&(g[0]==g[2]||g[1]==g[3])||n.Ko(t.R,e)==3&&g[0]==g[2])return  false;for(s=0,h=0,b=false,D=1;D<n.Ko(t.R,e);D++)if(g[D-1]==g[D]){if(A==1||g[D]==0)return  false;s=n.Yo(t.R,e,c[D-1]),h=n.Yo(t.R,e,c[D]),A==3&&n.fq(t.R,n.$o(t.R,e,c[D]))&&(t.W[e]=true),b=true;}return A!=1&&!b?false:(P=t.jb?n.yf(t,e,c):n.xf(t,e,c),A==1?t.db[e]=P:A==2&&(P==1?n.ah(t.e[s],Ct(t.f[e])):P==2&&n.ah(t.e[h],Ct(t.f[e]))),true);},n.xf=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h;for(h=F(k(l.d7,2),S1,7,0,[F(k(l.d7,1),H,6,15,[2,1,2,1]),F(k(l.d7,1),H,6,15,[1,2,2,1]),F(k(l.d7,1),H,6,15,[1,1,2,2]),F(k(l.d7,1),H,6,15,[2,1,1,2]),F(k(l.d7,1),H,6,15,[2,2,1,1]),F(k(l.d7,1),H,6,15,[1,2,1,2])]),P=z(l.b7,jt,6,n.Ko(t.R,e),15,1),D=0;D<n.Ko(t.R,e);D++)P[D]=n.Xl(t.R,n.Yo(t.R,e,A[D]),e);if(Q=n.dp(t.R,e,A,P,null)<<24>>24,Q!=3)return Q;for(L=0,s=0,u=0;u<n.Ko(t.R,e);u++)r=n.$o(t.R,e,A[u]),n.Yl(t.R,0,r)==e&&(n.hm(t.R,r)==129&&(s!=0&&n.Un(t.R,e),L=u,s=1),n.hm(t.R,r)==257&&(s!=0&&n.Un(t.R,e),L=u,s=2));if(s==0)return 3;for(i=1;i<n.Ko(t.R,e);i++)P[i]<P[0]&&(P[i]+=X1);if(n.Ko(t.R,e)==3){switch(L){case 0:(P[1]<P[2]&&P[2]-P[1]<n1||P[1]>P[2]&&P[1]-P[2]>n1)&&(s=3-s);break;case 1:P[2]-P[0]>n1&&(s=3-s);break;case 2:P[1]-P[0]<n1&&(s=3-s);}return s==1?2:1;}return o=0,P[1]<=P[2]&&P[2]<=P[3]?o=0:P[1]<=P[3]&&P[3]<=P[2]?o=1:P[2]<=P[1]&&P[1]<=P[3]?o=2:P[2]<=P[3]&&P[3]<=P[1]?o=3:P[3]<=P[1]&&P[1]<=P[2]?o=4:P[3]<=P[2]&&P[2]<=P[1]&&(o=5),h[o][L]==s?2:1;},n.yf=function(t,e,A){var P,r,i,D,u,o;for(P=z(l.d7,H,6,4,15,1),u=0;u<n.Ko(t.R,e);u++)P[u]=n.Yo(t.R,e,A[u]);for(n.Ko(t.R,e)==3&&(P[3]=e),r=h1(l.b7,[l1,jt],[15,6],15,[3,3],2),D=0;D<3;D++)r[D][0]=n.Pl(t.R,P[D+1])-n.Pl(t.R,P[0]),r[D][1]=n.Ql(t.R,P[D+1])-n.Ql(t.R,P[0]),r[D][2]=n.Rl(t.R,P[D+1])-n.Rl(t.R,P[0]);return o=z(l.b7,jt,6,3,15,1),o[0]=r[0][1]*r[1][2]-r[0][2]*r[1][1],o[1]=r[0][2]*r[1][0]-r[0][0]*r[1][2],o[2]=r[0][0]*r[1][1]-r[0][1]*r[1][0],i=(r[2][0]*o[0]+r[2][1]*o[1]+r[2][2]*o[2])/(j.Math.sqrt(r[2][0]*r[2][0]+r[2][1]*r[2][1]+r[2][2]*r[2][2])*j.Math.sqrt(o[0]*o[0]+o[1]*o[1]+o[2]*o[2])),i>0?1:2;},n.zf=function(t){var e,A;for(e=0,U.Yqb(t.e),A=0;A<t.e.length;A++)(A==0||n.bh(t.e[A],t.e[A-1])!=0)&&++e,t.f[t.e[A].a]=e;return e;},n.Af=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h;if(!t.A){for(t.A=new U.Lj(),L=0,s=z(l.d7,H,6,t.R.f,15,1),D=z(l.d7,H,6,t.R.f,15,1),o=z(l.d7,H,6,t.R.g,15,1),e=0;e<t.R.f;e++)if(s[e]==0&&(n.eq(t.R,e)||n.Po(t.R,e)==1)){for(D[0]=e,u=1,Q=0,s[e]=++L,A=z(l.zeb,ht,6,t.R.g,16,1),i=0;i<u;i++)for(h=0;h<n.Zo(t.R,D[i]);h++)r=n.$o(t.R,D[i],h),(n.fq(t.R,r)||n.em(t.R,r)==2||n.Up(t.R,r))&&(P=n.Yo(t.R,D[i],h),A[r]||(o[Q++]=r,A[r]=true),s[P]==0&&(D[u++]=P,s[P]=L));U.rj(t.A,new n.jh(D,u,o,Q));}}},n.Bf=function(t){var e;for(t.S=z(l.zeb,ht,6,t.R.f,16,1),e=0;e<t.R.f;e++)if(n.Sl(t.R,e)==7){if(n.Zo(t.R,e)==4){t.S[e]=true;continue;}if(n.Zo(t.R,e)==3){if(n.To(t.R,e)==3){t.S[e]=true;continue;}if(n.Al(t.R,e)==1){t.S[e]=true;continue;}if(n.Zp(t.R,e,true))continue;if((t.Q&32)!=0){t.S[e]=true;continue;}if(n.dq(t.R,e)){t.S[e]=true;continue;}}}},n.Cf=function(t,e){var A,P,r,i;for(r=false,P=0;P<t.R.g;P++)n.rf(t,P,1)&&(t.q[P]=t.L,e&&n.Rf(t,P),r=true);for(i=false,A=0;A<t.R.f;A++)n.wf(t,A,1)&&(t.ib[A]=t.L,e&&n.Sf(t,A),i=true);return i&&(t.L=!t.L),r||i;},n.Df=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K;for(C=z(l.zeb,ht,6,t.R.f,16,1),v=z(l.zeb,ht,6,t.R.g,16,1),e=0,S=false,(t.Q&128)!=0&&(t.Y=z(l.d7,H,6,t.R.f,15,1),t.X=z(l.d7,H,6,t.R.g,15,1)),P=0;P<t.R.f;P++)t.W[P]&&(t.fb[P]||n.wf(t,P,1)&&(t.fb[P]=true,C[P]=true,++e));for(i=0;i<t.R.g;i++)t.V[i]&&(t.p[i]||n.rf(t,i,1)&&(t.p[i]=true,v[i]=true,++e));if(e==1){for(A=0;A<t.R.f;A++)if(C[A]){t.db[A]=0;break;}for(r=0;r<t.R.g;r++)if(v[r]){t.o[r]=0;break;}}else if(e>1)for(n.Af(t),t.T=0,u=new U.wqb(t.A);u.a<u.c.a.length;){for(D=U.vqb(u),M=0,V=0,L=0,Q=0,s=-1,o=-1,c=0;c<D.a.length;c++)C[D.a[c]]&&(++M,(t.db[D.a[c]]==1||t.db[D.a[c]]==2)&&(++V,S=true,s<t.f[D.a[c]]&&(s=t.f[D.a[c]],L=D.a[c])));for(g=0;g<D.b.length;g++)v[D.b[g]]&&(++M,O=t.f[n.Yl(t.R,0,D.b[g])],K=t.f[n.Yl(t.R,1,D.b[g])],h=O>K?(O<<16)+K:(K<<16)+O,(t.o[D.b[g]]==1||t.o[D.b[g]]==2)&&(++V,S=true,o<h&&(o=h,Q=D.b[g])));if(M!=0)if(M==1){for(d=0;d<D.a.length;d++)C[D.a[d]]&&(t.db[D.a[d]]=0);for(b=0;b<D.b.length;b++)v[D.b[b]]&&(t.o[D.b[b]]=0);}else if(V==1){for(d=0;d<D.a.length;d++)C[D.a[d]]&&(t.db[D.a[d]]=3);for(b=0;b<D.b.length;b++)v[D.b[b]]&&(t.o[D.b[b]]=3);}else {if((t.Q&128)!=0){for(++t.T,d=0;d<D.a.length;d++)C[D.a[d]]&&(t.Y[D.a[d]]=t.T);for(b=0;b<D.b.length;b++)v[D.b[b]]&&(t.X[D.b[b]]=t.T);}if(m=false,s!=-1?t.db[L]==2&&(m=true):t.o[Q]==2&&(m=true),m){for(d=0;d<D.a.length;d++)if(C[D.a[d]])switch(t.db[D.a[d]]){case 1:t.db[D.a[d]]=2;break;case 2:t.db[D.a[d]]=1;}for(b=0;b<D.b.length;b++)if(v[D.b[b]])switch(t.o[D.b[b]]){case 1:t.o[D.b[b]]=2;break;case 2:t.o[D.b[b]]=1;}}}}return S;},n.Ef=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;for(i=h1(l.d7,[S1,H],[7,6],15,[2,32],2),D=0;D<2;D++){for(A=z(l.d7,S1,7,32,0,2),h=0,r=0;r<32;r++)if(e[D][r]!=null){for(L=e[D][r].length,A[r]=z(l.d7,H,6,L,15,1),u=0;u<L;u++)A[r][u]=t.f[e[D][r][u]];w.qzb(A[r],w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),++h;}for(s=h;s>0;s--){for(Q=0,o=null,P=0;P<32;P++)if(A[P]!=null){if(o==null||o.length<A[P].length)o=A[P],Q=P;else if(o.length==A[P].length){for(u=o.length-1;u>=0;u--)if(o[u]<A[P][u]){o=A[P],Q=P;break;}}}i[D][Q]=s,A[Q]=null;}}return i;},n.Ff=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v;if(s=false,t.R.K){for(Q=0;Q<t.R.g;Q++)if(n.gm(t.R,Q)!=0){s=true;break;}}for(t.O=2,A=0;A<t.R.f;A++)t.O=j.Math.max(t.O,n.Zo(t.R,A)+n.qp(t.R,A));for(o=j.Math.max(2,s?(62+t.c+t.O*(t.c+23))/63|0:(62+t.c+t.O*(t.c+5))/63|0),t.f=z(l.d7,H,6,t.R.q,15,1),t.e=z(l.p7,Mn,101,t.R.f,0,1),P=0;P<t.R.f;P++)t.e[P]=new n.eh(o);for(u=false,r=0;r<t.R.f;r++)n.dh(t.e[r],r),kt(G(n.Nl(t.R,r),1),0)||n.Il(t.R,r)!=null?n._g(t.e[r],8,6):n._g(t.e[r],8,Ct(n.Sl(t.R,r))),n._g(t.e[r],8,Ct(n.Ll(t.R,r))),n._g(t.e[r],2,Ct(n.Po(t.R,r))),n._g(t.e[r],4,Ct(n.Zo(t.R,r)+n.qp(t.R,r))),kt(G(n.Nl(t.R,r),1),0)?n._g(t.e[r],4,8):n._g(t.e[r],4,Ct(8+n.Al(t.R,r))),n._g(t.e[r],5,Ct(j.Math.min(31,n.To(t.R,r)))),n._g(t.e[r],4,Ct(n.uf(t,r)+1)),n._g(t.e[r],2,Ct(n.Ol(t.R,r)>>4)),t.R.K&&(n._g(t.e[r],51,n.Nl(t.R,r)),n.Il(t.R,r)!=null&&(u=true));if(t.U=n.Jf(t),t.U<t.R.f){for(e=0;e<t.R.f;e++){for(n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),h=z(l.d7,H,6,n.Zo(t.R,e),15,1),c=0;c<n.Zo(t.R,e);c++)h[c]=t.f[n.Yo(t.R,e,c)]<<5,h[c]|=j.Math.min(31,n.Xo(t.R,n.$o(t.R,e,c)));for(w.qzb(h,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),g=t.O;g>h.length;g--)n._g(t.e[e],t.c+5,0);for(b=h.length-1;b>=0;b--)n._g(t.e[e],t.c+5,l.Oeb(h[b]));}t.U=n.Jf(t);}if(u&&t.U<t.R.f){for(e=0;e<t.R.f;e++){for(n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),D=n.Il(t.R,e),C=D==null?0:j.Math.min(12,D.length),c=12;c>C;c--)n._g(t.e[e],8,0);for(b=C-1;b>=0;b--)n._g(t.e[e],8,l.Oeb(D[b]));}t.U=n.Jf(t);}if(s&&t.U<t.R.f){for(e=0;e<t.R.f;e++){for(n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),L=z(l.e7,n0,6,n.Zo(t.R,e)+n.qp(t.R,e),14,1),d=0,c=0;c<n.Lo(t.R,e);c++)(c<n.Zo(t.R,e)||c>=n.Ko(t.R,e))&&(L[d]=Ct(t.f[n.Yo(t.R,e,c)]),L[d]=b1(L[d],23),L[d]=Pt(L[d],Ct(n.gm(t.R,n.$o(t.R,e,c)))),++d);for(w.qzb(L,w.Xzb(h0(U.drb.prototype.kd,U.drb,[]))),g=t.O;g>L.length;g--)n._g(t.e[e],t.c+23,0);for(b=L.length-1;b>=0;b--)n._g(t.e[e],t.c+23,L[b]);}t.U=n.Jf(t);}if((t.Q&8)!=0&&t.U<t.R.f){for(m=new n.Yv(),i=0;i<t.R.f;i++)n.Dl(t.R,i)!=null&&n.Wv(m,n.Dl(t.R,i));for(e=0;e<t.R.f;e++)v=n.Dl(t.R,e)==null?0:1+n.Xv(m,n.Dl(t.R,e)),n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),n._g(t.e[e],t.c,l.Oeb(v));t.U=n.Jf(t);}if((t.Q&16)!=0&&t.U<t.R.f){for(e=0;e<t.R.f;e++)n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),n._g(t.e[e],1,Ct(n.Om(t.R,e)?1:0));t.U=n.Jf(t);}(t.Q&512)!=0&&t.R.K&&n.hf(t);},n.Gf=function(t){var e,A,P,r,i,D,u,o,Q;for(A=0;A<t.R.f;A++)n.dh(t.e[A],A),n._g(t.e[A],2*t.c+4,b1(Ct(t.f[A]),t.c+4));for(o=1;o<=t.U;o++){for(r=false,P=0;P<t.R.f;P++)t.f[P]==o&&(r=r|n.wf(t,P,2));if(r){if(u=t.U,t.U=n.Jf(t),t.U!=u)return  true;for(e=0;e<t.R.f;e++)n.dh(t.e[e],e),n._g(t.e[e],2*t.c+4,b1(Ct(t.f[e]),t.c+4));}}for(Q=z(l.q7,Mn,172,t.R.g,0,1),D=0;D<Q.length;D++)Q[D]=new n.hh(t.f[n.Yl(t.R,0,D)],t.f[n.Yl(t.R,1,D)],D);for(U.Wqb(Q,0,Q.length,null),i=0;i<Q.length;i++)if(n.rf(t,Q[i].a,2)){for(;i+1<Q.length&&n.gh(Q[i],Q[i+1])==0;)n.rf(t,Q[++i].a,2);if(u=t.U,t.U=n.Jf(t),t.U!=u)return  true;for(e=0;e<t.R.f;e++)n.dh(t.e[e],e),n._g(t.e[e],2*t.c+4,b1(Ct(t.f[e]),t.c+4));}return  false;},n.Hf=function(t){var e,A,P,r,i,D,u,o,Q,L;for(i=0,A=0;A<t.R.f;A++)t.bb[A]!=0&&(t.bb[A]!=2||(t.Q&256)==0)&&++i;if(i!=0){for(L=z(l.d7,H,6,i,15,1),i=0,P=0;P<t.R.f;P++)t.bb[P]!=0&&(t.bb[P]!=2||(t.Q&256)==0)&&(L[i]=t.bb[P]<<29|t.ab[P]<<24|t.f[P]<<12|P,++i);for(w.qzb(L,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),D=0,Q=0,u=L[0]&O0;;)if(++Q,Q==L.length||u!=(L[Q]&O0)){for(r=z(l.d7,H,6,Q-D,15,1),o=D;o<Q;o++)e=L[o]&4095,r[o-D]=e,t.gb[e]=true;if(U.rj(t.hb,r),Q==L.length)break;u=L[Q]&O0,D=Q;}}},n.If=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b;for(u=false,o=0;o<t.hb.a.length;o++){for(D=U.wj(t.hb,o),e=true,b=-1,Q=false,s=0;s<D.length;s++){if(A=D[s],t.db[A]==0){e=false;break;}if(t.db[A]!=3){for(L=true,h=0;h<D.length;h++)if(h!=s&&t.f[A]==t.f[D[h]]){L=false;break;}L&&b<t.f[A]&&(b=t.f[A],Q=t.db[A]==1);}}if(e&&b!=-1){for(P=D,r=0,i=P.length;r<i;++r)A=P[r],(t.db[A]==1||t.db[A]==2)&&(t.eb[A]=Q),t.gb[A]=false;U.Ej(t.hb,D),u=true,--o;}}return u;},n.Jf=function(t){var e,A;e=n.zf(t);do A=e,n.vf(t),e=n.zf(t);while(A!=e);return e;},n.Kf=function(t){for((t.Q&1)!=0&&(t.Q&2)==0&&(t.g=U.Eqb(t.f,t.R.f)),(t.Q&z1)==0&&(t.W=z(l.zeb,ht,6,t.R.f,16,1),t.V=z(l.zeb,ht,6,t.R.g,16,1),t.U<t.R.f&&(n.jf(t),(t.Q&z1)==0&&(n.If(t),t.P&&n.th(t.P,t.f)))),t.g==null&&(t.Q&1)!=0&&(t.Q&2)!=0&&(t.g=U.Eqb(t.f,t.R.f));t.U<t.R.f;)n.kf(t),(t.Q&z1)==0&&(n.If(t),t.P&&n.th(t.P,t.f));(t.Q&z1)==0&&(n.If(t),n.Df(t),n.gg(t));},n.Lf=function(t){var e,A,P,r,i;for(i=t.U,r=U.Eqb(t.f,t.R.f),t.R.K||(n.Nf(t),n.Bg(t,i,r)),t.bb=z(l._6,J,6,t.R.f,15,1),t.ab=z(l._6,J,6,t.R.f,15,1),A=0;A<t.R.f;A++)t.bb[A]=n.Gl(t.R,A)<<24>>24,t.ab[A]=n.Fl(t.R,A)<<24>>24;for(t.n=z(l._6,J,6,t.R.g,15,1),t.k=z(l._6,J,6,t.R.g,15,1),P=0;P<t.R.g;P++)t.n[P]=n.bm(t.R,P)<<24>>24,t.k[P]=n.am(t.R,P)<<24>>24;for(n.Mf(t),t.Z=false,t.M=z(l.zeb,ht,6,t.R.f,16,1),e=0;e<t.R.f;e++)t.db[e]!=0&&(t.M[e]=true,t.Z=true);n.Pf(t),t.P=null,t.cb=z(l.zeb,ht,6,t.R.f,16,1),t.Z&&(t.P=new n.zh(t.R,r,t.M,t.db,t.o,t.bb,t.ab,t.ib,t.q,t.cb),n.uh(t.P)),t.eb=z(l.zeb,ht,6,t.R.f,16,1),t.gb=z(l.zeb,ht,6,t.R.f,16,1),t.hb=new U.Lj(),t.P&&n.th(t.P,t.f),n.Hf(t),n.Bg(t,i,r),n.Of(t),t.P&&(t.K=n.qh(t.P)),n.Zf(t);},n.Mf=function(t){var e,A,P,r,i,D;for(t.L=true,i=n.Cf(t,false);t.U<t.R.f&&i;){for(e=0;e<t.R.f;e++)n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),D=t.db[e]<<7,(t.db[e]==1||t.db[e]==2)&&t.bb[e]!=0&&(D|=t.bb[e]<<5,D|=t.ab[e]),n._g(t.e[e],18,b1(l.Oeb(D),9));for(A=0;A<t.R.g;A++)P=t.o[A]<<7,(t.o[A]==1||t.o[A]==2)&&n.hm(t.R,A)==1&&t.n[A]!=0&&(P|=t.n[A]<<5,P|=t.k[A]),n.ah(t.e[n.Yl(t.R,0,A)],l.Oeb(P)),n.ah(t.e[n.Yl(t.R,1,A)],l.Oeb(P));if(r=n.Jf(t),t.U==r)break;t.U=r,i=n.Cf(t,false);}},n.Nf=function(t){var e,A,P,r;for(t.L=true,t.$=z(l._6,J,6,t.R.f,15,1),t.i=z(l._6,J,6,t.R.g,15,1),r=n.Cf(t,true);t.U<t.R.f&&r;){for(e=0;e<t.R.f;e++)n.dh(t.e[e],e),n._g(t.e[e],t.c+4,Pt(b1(Ct(t.f[e]),4),Ct(t.db[e]<<2)));for(A=0;A<t.R.g;A++)n.ah(t.e[n.Yl(t.R,0,A)],Ct(t.o[A])),n.ah(t.e[n.Yl(t.R,1,A)],Ct(t.o[A]));if(P=n.Jf(t),t.U==P)break;t.U=P,r=n.Cf(t,true);}},n.Of=function(t){var e,A,P,r,i,D,u;for(t.L=true,P=n.Yf(t),t.P&&n.th(t.P,t.f)&&(P=n.Yf(t)),n.Cf(t,false)&&n.If(t),D=true;t.U<t.R.f&&D;){for(r=n.Ef(t,P),e=0;e<t.R.f;e++)n.dh(t.e[e],e),n._g(t.e[e],t.c,Ct(t.f[e])),n._g(t.e[e],20,0),!t.cb[e]&&t.bb[e]!=0&&n.ah(t.e[e],B1(Ct(t.bb[e]<<18),b1(Ct(r[t.bb[e]==1?0:1][t.ab[e]]),8))),u=t.db[e],t.eb[e]&&(u==1?u=2:u==2&&(u=1)),n.ah(t.e[e],l.Oeb(u<<4));for(A=0;A<t.R.g;A++)n.ah(t.e[n.Yl(t.R,0,A)],Ct(t.o[A])),n.ah(t.e[n.Yl(t.R,1,A)],Ct(t.o[A]));if(i=n.Jf(t),t.U==i)break;t.U=i,D=false,t.P&&n.th(t.P,t.f)&&(D=true,P=n.Yf(t)),n.Cf(t,false)&&(D=true,n.If(t));}},n.Pf=function(t){var e,A;for(e=0;e<t.R.f;e++)(!t.M[e]||t.db[e]==3)&&(t.bb[e]=0);for(A=0;A<t.R.g;A++)(n.hm(t.R,A)!=1||t.o[A]==0||t.o[A]==3)&&(t.n[A]=0);},n.Qf=function(t,e,A){t.a==null&&(t.a=z(l._6,J,6,t.R.f,15,1),U.Hqb(t.a)),t.a[e]=A<<24>>24;},n.Rf=function(t,e){var A,P,r,i,D,u;if((t.o[e]==1||t.o[e]==2)&&!n.iq(t.R,e)){D=false;try{for(i=0;i<2;i++)if(A=n.Yl(t.R,i,e),n.Zo(t.R,A)==3){for(P=z(l.d7,H,6,2,15,1),r=0,u=0;u<n.Zo(t.R,A);u++)n.$o(t.R,A,u)!=e&&(P[r++]=n.Yo(t.R,A,u));t.f[P[0]]>t.f[P[1]]^n.Tf(t,A,P[0],P[1])&&(D=!D);}}catch(o){if(o=H1(o),Jt(o,19)){t.i[e]=3;return;}else throw Ut(o);}t.o[e]==1^D?t.i[e]=1:t.i[e]=2;}},n.Sf=function(t,e){var A,P,r,i,D,u,o;if(t.db[e]==1||t.db[e]==2){if(u=false,n.Po(t.R,e)==2&&n.Zo(t.R,e)==2)try{for(D=0;D<2;D++)if(A=n.Yo(t.R,e,D),n.Zo(t.R,A)==3){for(r=z(l.d7,H,6,2,15,1),i=0,o=0;o<n.Zo(t.R,A);o++)n._o(t.R,A,o)==1&&(r[i++]=n.Yo(t.R,A,o));t.f[r[0]]>t.f[r[1]]^n.Tf(t,A,r[0],r[1])&&(u=!u);}}catch(Q){if(Q=H1(Q),Jt(Q,19)){t.$[e]=3;return;}else throw Ut(Q);}else {try{P=n.Vf(t,e);}catch(Q){if(Q=H1(Q),Jt(Q,19)){t.$[e]=3;return;}else throw Ut(Q);}for(D=1;D<P.length;D++)for(o=0;o<D;o++)t.f[P[D]]<t.f[P[o]]&&(u=!u);}t.db[e]==1^u?t.$[e]=1:t.$[e]=2;}},n.Tf=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt;if(n.Sl(t.R,A)!=n.Sl(t.R,P))return n.Sl(t.R,A)>n.Sl(t.R,P);if(n.Ll(t.R,A)!=n.Ll(t.R,P))return Dt=n.Lm(t.R,A)?(n.Qk(),n.Ok)[n.Sl(t.R,A)]:n.Ll(t.R,A),Lt=n.Lm(t.R,P)?(n.Qk(),n.Ok)[n.Sl(t.R,P)]:n.Ll(t.R,P),Dt>Lt;for(V=t.R.f,C=z(l.d7,H,6,V,15,1),M=z(l.d7,H,6,V,15,1),S=z(l.d7,H,6,V,15,1),v=z(l.zeb,ht,6,V,16,1),o=z(l.zeb,ht,6,t.R.q,16,1),C[0]=e,C[1]=A,C[2]=P,M[0]=-1,M[1]=0,M[2]=0,o[e]=true,o[A]=true,o[P]=true,h=1,O=2,Qt=z(l.d7,H,6,64,15,1),Qt[1]=1,Qt[2]=3,c=2;h<=O;){for(;h<Qt[c];){if(b=C[h],!v[h]){for(g=0,d=0,N=0;N<n.Zo(t.R,b);N++){if(L=n.Yo(t.R,b,N),O+n._o(t.R,b,N)+1>=V&&(V+=t.R.f,C=n.Fg(C,V),M=n.Fg(M,V),S=n.Fg(S,V),v=n.Gg(v,V)),n.Xp(t.R,n.$o(t.R,b,N)))++g,d+=n.Sl(t.R,L);else if(L!=e)for(ut=1;ut<n._o(t.R,b,N);ut++)++O,C[O]=L,M[O]=h,v[O]=true;if(rt=M[h],L!=C[rt]){if(u=false,o[L])for(ot=M[rt];ot!=-1;){if(L==C[ot]){u=true;break;}ot=M[ot];}u?(++O,C[O]=L,M[O]=h,v[O]=true):(++O,C[O]=L,M[O]=h,o[L]=true);}}g!=0&&(++O,S[O]=(d<<2)/g|0,M[O]=h,v[O]=true);}if(++h,h==Y0)throw Ut(new T.bC("Emergency break in while loop."));}for(Qt.length==c+1&&(Qt=n.Fg(Qt,Qt.length+64)),Qt[c+1]=O+1,K=Qt[c];K<Qt[c+1];K++)S[K]==0&&(S[K]=(n.Sl(t.R,C[K])==151||n.Sl(t.R,C[K])==152?1:n.Sl(t.R,C[K]))<<2),S[K]+=S[M[K]]<<16;if(n.Xf(t,v,S,M,C,Qt,c),S[1]!=S[2])return S[1]>S[2];c>1&&n.Uf(S,M,Qt,c),++c;}for(s=z(l.d7,H,6,t.R.f,15,1),I=false,i=0;i<t.R.f;i++)if(o[i]&&!n.Lm(t.R,i)){I=true;break;}if(I){for(D=0;D<t.R.f;D++)s[D]=n.Lm(t.R,D)?(n.Qk(),n.Ok)[n.Sl(t.R,D)]:n.Ll(t.R,D);if(n.Wf(t,v,S,M,C,s,Qt,c))return S[1]>S[2];}for(U.Oqb(s,s.length,0),m=false,Q=0;Q<t.R.g;Q++)(o[n.Yl(t.R,0,Q)]||o[n.Yl(t.R,1,Q)])&&(t.i[Q]==1?(s[n.Yl(t.R,0,Q)]=1,s[n.Yl(t.R,1,Q)]=1,m=true):t.i[Q]==2&&(s[n.Yl(t.R,0,Q)]=2,s[n.Yl(t.R,1,Q)]=2,m=true));if(m&&n.Wf(t,v,S,M,C,s,Qt,c))return S[1]>S[2];for(U.Oqb(s,s.length,0),yt=false,r=0;r<t.R.f;r++)o[r]&&(t.$[r]==2?(s[r]=1,yt=true):t.$[r]==1&&(s[r]=2,yt=true));if(yt&&n.Wf(t,v,S,M,C,s,Qt,c))return S[1]>S[2];throw t.d=true,Ut(new T.bC("no distinction applying CIP rules"));},n.Uf=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h;for(s=A[P],D=A[P+1]-s,h=z(l.n7,C1,174,D,0,1),o=0;o<D;o++)h[o]=new n.$g(),h[o].c=t[o+s],h[o].b=e[o+s],h[o].a=o+s;for(r=new n.Xg(),L=P;L>1;L--){for(Q=0;Q<D;Q++)h[Q].c+=t[h[Q].b]<<16,h[Q].b=e[h[Q].b];for(U.Wqb(h,0,h.length,r),i=1,u=0;u<D;u++)t[h[u].a]=i,u!=D-1&&n.Wg(h[u],h[u+1])!=0&&++i;}},n.Vf=function(t,e){var A,P,r,i,D,u,o;for(D=n.Ko(t.R,e),u=z(l.d7,H,6,D,15,1),r=0;r<D;r++)u[r]=n.Yo(t.R,e,r);for(P=D;P>1;P--){for(A=false,i=1;i<P;i++)n.Tf(t,e,u[i-1],u[i])&&(A=true,o=u[i-1],u[i-1]=u[i],u[i]=o);if(!A)break;}return u;},n.Wf=function(t,e,A,P,r,i,D,u){var o,Q;for(Q=1;Q<u;Q++){for(o=D[Q];o<D[Q+1];o++)A[o]=i[r[o]]+(A[P[o]]<<8);if(n.Xf(t,e,A,P,r,D,Q),A[1]!=A[2])return  true;Q>1&&n.Uf(A,P,D,Q);}return  false;},n.Xf=function(t,e,A,P,r,i,D){var u,o,Q,L,s,h,b,c,g,d,m;for(s=D;s>1;s--){for(g=i[s]-i[s-1],m=z(l.l7,C1,173,g,0,1),u=i[s],c=0;c<g;c++){for(d=i[s-1]+c,h=u;h<i[s+1]&&P[h]==d;)++h;for(m[c]=new n.Ug(),m[c].c=d,m[c].d=A[d],m[c].b=e[d]?0:n.Bp(t.R,r[d]),m[c].a=z(l.d7,H,6,h-u,15,1),L=u;L<h;L++)m[c].a[L-u]=A[L];w.qzb(m[c].a,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),u=h;}for(o=new n.Rg(),U.Wqb(m,0,m.length,o),Q=1,b=0;b<g;b++)A[m[b].c]=Q,b!=g-1&&n.Qg(m[b],m[b+1])!=0&&++Q;}},n.Yf=function(t){var e,A;for(A=h1(l.d7,[c0,S1],[28,7],0,[2,32],2),e=0;e<t.R.f;e++)t.M[e]&&(t.bb[e]==1?A[0][t.ab[e]]=n.Ah(A[0][t.ab[e]],e):t.bb[e]==2&&(A[1][t.ab[e]]=n.Ah(A[0][t.ab[e]],e)));return A;},n.Zf=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h;for(i=0,L=0,D=0,u=0,o=0,Q=0,s=0,h=false,e=z(l.zeb,ht,6,32,16,1),A=0;A<t.R.f;A++)t.db[A]!=0&&(++i,t.db[A]==3?++L:t.bb[A]==0?(++D,t.P&&n.ph(t.P,A)&&++u):t.bb[A]==2?t.ab[A]==0&&++Q:t.bb[A]==1&&(r=t.ab[A],e[r]||(++s,e[r]=true),t.ab[A]==0&&++o,t.P&&n.ph(t.P,A)&&(h=true)));for(P=0;P<t.R.g;P++)t.o[P]!=0&&n.hm(t.R,P)==1&&(++i,t.o[P]==3?++L:t.n[P]==0?(++D,t.P&&n.ph(t.P,n.Yl(t.R,0,P))&&n.ph(t.P,n.Yl(t.R,1,P))&&++u):t.n[P]==2?t.k[P]==0&&++Q:t.n[P]==1&&(r=t.k[P],e[r]||(++s,e[r]=true),t.k[P]==0&&++o,t.P&&n.ph(t.P,n.Yl(t.R,0,P))&&n.ph(t.P,n.Yl(t.R,1,P))&&(h=true)));if(i==0){n.Mn(t.R,q1);return;}if(L!=0){n.Mn(t.R,0);return;}if(t.K){n.Mn(t.R,A0+(1<<s));return;}o+u==i&&!h?n.Mn(t.R,JA):D==i?n.Mn(t.R,Ue):Q==i?n.Mn(t.R,327680):D==i-1&&o==1?n.Mn(t.R,tn):n.Mn(t.R,458752+(1<<s));},n.$f=function(t,e,A){for(;A!=0;)t.t==0&&((!t.r||t.v!=63)&&(t.v+=64),T.Jjb(t.u,t.v&r1),t.t=6,t.v=0),t.v<<=1,t.v=Gt(Pt(Ct(t.v),G(e,1))),e=t0(e,1),--A,--t.t;},n._f=function(t){return t.v<<=t.t,(!t.r||t.v!=63)&&(t.v+=64),T.Jjb(t.u,t.v&r1),t.u.a;},n.ag=function(t,e){t.u=new T.Pjb(),t.t=6,t.v=0,t.r=e;},n.bg=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d;if(t.R.f==0){t.s="";return;}for(s=t.jb&t.b,d=t.jb?16:8,n.ag(t,true),T.Jjb(t.u,s?35:33),n.$f(t,Ct(t.jb?1:0),1),n.$f(t,l.Oeb(e?1:0),1),n.$f(t,l.Oeb(d>>1),4),c=0,o=1;o<t.R.f;o++)c=n.rg(t,t.B[o],t.F[o]==-1?-1:t.B[t.F[o]],c,A);if(s)for(Q=0;Q<t.R.f;Q++)for(P=t.B[Q],b=n.Zo(t.R,P);b<n.Ko(t.R,P);b++)c=n.rg(t,n.Yo(t.R,P,b),P,c,A);if(t.R.f>1&&c==0){t.s="";return;}for(D=1<<d,h=c/((D>>1)-1),g=c+.5*h,L=1;L<t.R.f;L++)n.cg(t,t.B[L],t.F[L]==-1?-1:t.B[t.F[L]],g,h,d,A);if(s)for(u=0;u<t.R.f;u++)for(P=t.B[u],b=n.Zo(t.R,P);b<n.Ko(t.R,P);b++)n.cg(t,n.Yo(t.R,P,b),P,g,h,d,A);e&&(i=t.jb?1.5:(n.Qk(),n.Qk(),n.Pk),r=n.Wl(t.R,s?t.R.q:t.R.f,s?t.R.r:t.R.g,i,A),n.$f(t,Ct(j.Math.min(D-1,j.Math.max(0,Ft(.5+j.Math.log(r/.1)*j.Math.LOG10E/(j.Math.log(2e3)*j.Math.LOG10E)*(D-1))))),d),n.$f(t,Ct(n.fg(A[t.B[0]].a/r,D)),d),n.$f(t,Ct(n.fg(A[t.B[0]].b/r,D)),d),t.jb&&n.$f(t,Ct(n.fg(A[t.B[0]].c/r,D)),d)),t.s=n._f(t);},n.cg=function(t,e,A,P,r,i,D){var u,o,Q;u=A==-1?(D[e].a-D[t.B[0]].a)/8:D[e].a-D[A].a,o=A==-1?(D[e].b-D[t.B[0]].b)/8:D[e].b-D[A].b,n.$f(t,Ct(Ft((P+u)/r)),i),n.$f(t,Ct(Ft((P+o)/r)),i),t.jb&&(Q=A==-1?(D[e].c-D[t.B[0]].c)/8:D[e].c-D[A].c,n.$f(t,Ct(Ft((P+Q)/r)),i));},n.dg=function(t,e){var A;for(A=0;A<t.w;A++)e-=16;for(e<0&&T.Ujb();e>15;)n.$f(t,1,1),n.$f(t,15,4),e-=16,++t.w;n.$f(t,1,1),n.$f(t,l.Oeb(e),4);},n.eg=function(t){var e,A,P,r,i,D;if(t.R.f==0){t.N="";return;}for(i=0,P=false,r=false,A=0;A<t.R.f;A++)i<n.Kl(t.R,A)&&(i=n.Kl(t.R,A)),n.zm(t.R,A)?P=true:r=true;if(i==0){t.N="";return;}for(D=n.Og(i),n.ag(t,true),n.$f(t,l.Oeb(D),4),n.$f(t,l.Oeb(P?1:0),1),n.$f(t,l.Oeb(r?1:0),1),e=0;e<t.R.f;e++)n.$f(t,Ct(n.Kl(t.R,t.B[e])),D),P&&r&&n.$f(t,Ct(n.zm(t.R,t.B[e])?1:0),1);t.N=n._f(t);},n.fg=function(t,e){var A,P,r,i;return A=e/2|0,r=t<0,t=j.Math.abs(t),i=e/32|0,P=j.Math.min(A-1,Gt(w1(j.Math.round(t*A/(t+i))))),r?A+P:P;},n.gg=function(t){var e,A,P;for(e=0;e<t.R.f;e++)t.db[e]==3&&!n.vm(t.R,e)&&n.Un(t.R,e),(n.Gl(t.R,e)==1||n.Gl(t.R,e)==2)&&t.db[e]==3&&n.Un(t.R,e),n.vm(t.R,e)&&t.db[e]!=3&&!n.Dg(t,e)&&n.Un(t.R,e);for(P=0;P<t.R.r;P++)n.Qm(t.R,P)&&!n.Cg(t,P)&&n.Un(t.R,n.Yl(t.R,0,P));for(A=0;A<t.R.g;A++)n.em(t.R,A)==2&&(n.Fm(t.R,A)&&(t.o[A]==1||t.o[A]==2)&&(t.o[A]=3,n.Ln(t.R,A,386)),t.o[A]==3&&!t.p[A]&&n.hm(t.R,A)!=386&&(n.Un(t.R,n.Yl(t.R,0,A)),n.Un(t.R,n.Yl(t.R,1,A)))),n.hm(t.R,A)==1&&t.o[A]==3&&!n.vm(t.R,n.Yl(t.R,0,A))&&!n.vm(t.R,n.Yl(t.R,1,A))&&(n.Un(t.R,n.Yl(t.R,0,A)),n.Un(t.R,n.Yl(t.R,1,A))),(n.bm(t.R,A)==1||n.bm(t.R,A)==2)&&(n.hm(t.R,A)!=1||t.o[A]!=1&&t.o[A]!=2)&&(n.Un(t.R,n.Yl(t.R,0,A)),n.Un(t.R,n.Yl(t.R,1,A)));},n.hg=function(t){var e,A,P,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;if(t.R.f!=0&&!t.G){for(t.I=0,S=0,A=1;A<t.R.f;A++)t.f[A]>t.f[S]&&(S=A);for(P=z(l.zeb,ht,6,t.R.f,16,1),D=z(l.zeb,ht,6,t.R.g,16,1),t.H=z(l.d7,H,6,t.R.f,15,1),t.B=z(l.d7,H,6,t.R.f,15,1),t.F=z(l.d7,H,6,t.R.f,15,1),t.C=z(l.d7,H,6,t.R.g,15,1),t.B[0]=S,t.H[S]=0,P[S]=true,o=0,Q=1,L=0;o<t.R.f;)if(o<Q){for(;;){for(c=0,g=0,h=-1,e=t.B[o],d=0;d<n.Lo(t.R,e);d++)(d<n.Zo(t.R,e)||d>=n.Ko(t.R,e))&&(u=n.Yo(t.R,e,d),!P[u]&&t.f[u]>h&&(c=u,g=n.$o(t.R,e,d),h=t.f[u]));if(h==-1)break;t.H[c]=Q,t.F[Q]=o,t.B[Q++]=c,t.C[L++]=g,P[c]=true,D[g]=true;}++o;}else {for(b=0,h=-1,e=0;e<t.R.f;e++)!P[e]&&t.f[e]>h&&(b=e,h=t.f[e]);t.H[b]=Q,t.F[Q]=-1,t.B[Q++]=b,P[b]=true;}for(t.D=z(l.d7,H,6,2*(t.R.g-L),15,1);;){for(C=t.R.M,v=t.R.M,M=-1,i=0;i<t.R.g;i++)D[i]||(t.H[n.Yl(t.R,0,i)]<t.H[n.Yl(t.R,1,i)]?(m=t.H[n.Yl(t.R,0,i)],s=t.H[n.Yl(t.R,1,i)]):(m=t.H[n.Yl(t.R,1,i)],s=t.H[n.Yl(t.R,0,i)]),(m<C||m==C&&s<v)&&(C=m,v=s,M=i));if(M==-1)break;D[M]=true,t.C[L++]=M,t.D[2*t.I]=C,t.D[2*t.I+1]=v,++t.I;}t.G=true;}},n.ig=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c;for(o=null,c=n.Dp(t.R),s=0;s<c.j.a.length;s++)if(c.f[s]){for(i=0,h=U.wj(c.j,s),A=h,P=0,r=A.length;P<r;++P)e=A[P],n.xg(t,e)&&++i;if(i!=0)if(b=U.wj(c.k,s),o==null&&(o=z(l.zeb,ht,6,t.R.g,16,1)),i==h.length){for(Q=-1,L=St,D=0;D<h.length;D++)L>t.B[b[D]]&&(L=t.B[b[D]],Q=D);for(;i>0;)o[b[Q]]=true,Q=n.Lg(Q+2,h.length),i-=2;}else {for(u=0;n.xg(t,h[u]);)++u;for(;!n.xg(t,h[u]);)u=n.Lg(u+1,h.length);for(;i>0;)for(o[b[u]]=true,u=n.Lg(u+2,h.length),i-=2;!n.xg(t,h[u]);)u=n.Lg(u+1,h.length);}}return o;},n.jg=function(t,e){var A,P,r,i,D,u,o,Q,L;for(n.hg(t),Q=new n.tw(t.R.f,t.R.g),n.Nn(Q,t.R.K),D=0;D<t.R.f;D++)n.kl(t.R,Q,t.B[D],0,0),n.nn(Q,D,t.bb[t.B[D]],t.ab[t.B[D]]);for(u=0;u<t.R.g;u++)n.ml(t.R,Q,t.C[u],0,0,t.H,false),!(Q.H[u]==257||Q.H[u]==129)&&Q.D[0][u]>Q.D[1][u]&&(L=Q.D[0][u],n.Cn(Q,0,u,Q.D[1][u]),Q.D[1][u]=L,Q.T=0),n.Fn(Q,u,t.n[t.C[u]],t.k[t.C[u]]);if(e)for(i=0;i<t.R.f;i++)for(A=t.B[i],o=n.Zo(t.R,A);o<n.Ko(t.R,A);o++)r=n.kl(t.R,Q,n.Yo(t.R,A,o),0,0),n.ll(t.R,Q,n.$o(t.R,A,o),0,0,t.H[A],r,false);for(P=0;P<Q.r;P++)A=Q.D[0][P],t.eb[t.B[A]]&&(Q.H[P]==257?(Q.H[P]=129,Q.T=0):Q.H[P]==129&&(Q.H[P]=257,Q.T=0));return n._v(t.R,Q),Q.T&=-9,Q;},n.kg=function(t,e){return t.o[e];},n.lg=function(t){return n.mg(t,t.jb);},n.mg=function(t,e){return t.s==null&&(n.hg(t),n.bg(t,e,t.R.J)),t.s;},n.ng=function(t){return t.N==null&&(n.hg(t),n.eg(t)),t.N;},n.og=function(t){return n.hg(t),t.B;},n.pg=function(t){return n.hg(t),t.H;},n.qg=function(t){return t.J==null&&(n.hg(t),(t.Q&z1)==0&&(n.zg(t),n.Ag(t,1),n.Ag(t,2)),n.yg(t)),t.J;},n.rg=function(t,e,A,P,r){var i,D,u;return i=A==-1?j.Math.abs(r[e].a-r[t.B[0]].a)/8:j.Math.abs(r[e].a-r[A].a),P<i&&(P=i),D=A==-1?j.Math.abs(r[e].b-r[t.B[0]].b)/8:j.Math.abs(r[e].b-r[A].b),P<D&&(P=D),t.jb&&(u=A==-1?j.Math.abs(r[e].c-r[t.B[0]].c)/8:j.Math.abs(r[e].c-r[A].c),P<u&&(P=u)),P;},n.sg=function(t,e){return t.X[e];},n.tg=function(t,e){return t.Y[e];},n.ug=function(t,e){return t.g==null?-1:t.g[e];},n.vg=function(t,e){return t.db[e];},n.wg=function(t,e){var A,P,r,i,D,u,o;for(o=n.Dp(t.R),A=0;A<o.j.a.length;A++)if(o.e[A]&&n.pt(o,A,e)){for(r=U.wj(o.j,A),i=0,D=r.length;i<D;++i)if(P=r[i],P!=e){for(u=0;u<n.Zo(t.R,P);u++)if(n.Up(t.R,n.$o(t.R,P,u)))return  true;}return  false;}return  false;},n.xg=function(t,e){var A,P,r;if(n.Po(t.R,e)<2)return  false;if(n.Zo(t.R,e)==2)return  true;for(A=0,r=0;r<n.Zo(t.R,e);r++)P=n.$o(t.R,e,r),n.Tp(t.R,P)&&(A+=n.em(t.R,P)-1);return A>1;},n.yg=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt,mt,y1,u1,Mt,Zt;if(n.ag(t,false),n.$f(t,9,4),mt=j.Math.max(n.Og(t.R.f),n.Og(t.R.g)),n.$f(t,l.Oeb(mt),4),mt==0){n.$f(t,Ct(t.R.K?1:0),1),n.$f(t,0,1),t.J=n._f(t);return;}for(y1=Mt=u1=Dt=0,o=0;o<t.R.f;o++)if(M1(G(n.Nl(t.R,o),1),0)){switch(n.Sl(t.R,o)){case 6:break;case 7:++y1;break;case 8:++Mt;break;default:++u1;}n.Al(t.R,o)!=0&&++Dt;}for(n.$f(t,Ct(t.R.f),mt),n.$f(t,Ct(t.R.g),mt),n.$f(t,l.Oeb(y1),mt),n.$f(t,l.Oeb(Mt),mt),n.$f(t,l.Oeb(u1),mt),n.$f(t,l.Oeb(Dt),mt),Q=0;Q<t.R.f;Q++)n.Sl(t.R,t.B[Q])==7&&M1(G(n.Nl(t.R,t.B[Q]),1),0)&&n.$f(t,l.Oeb(Q),mt);for(h=0;h<t.R.f;h++)n.Sl(t.R,t.B[h])==8&&M1(G(n.Nl(t.R,t.B[h]),1),0)&&n.$f(t,l.Oeb(h),mt);for(b=0;b<t.R.f;b++)n.Sl(t.R,t.B[b])!=6&&n.Sl(t.R,t.B[b])!=7&&n.Sl(t.R,t.B[b])!=8&&M1(G(n.Nl(t.R,t.B[b]),1),0)&&(n.$f(t,l.Oeb(b),mt),n.$f(t,Ct(n.Sl(t.R,t.B[b])),8));for(c=0;c<t.R.f;c++)n.Al(t.R,t.B[c])!=0&&M1(G(n.Nl(t.R,t.B[c]),1),0)&&(n.$f(t,l.Oeb(c),mt),n.$f(t,Ct(8+n.Al(t.R,t.B[c])),4));for(pt=0,S=0,g=1;g<t.R.f;g++)t.F[g]==-1?it=0:(it=1+t.F[g]-S,S=t.F[g]),pt<it&&(pt=it);for(bt=n.Og(pt),n.$f(t,l.Oeb(bt),4),S=0,d=1;d<t.R.f;d++)t.F[d]==-1?it=0:(it=1+t.F[d]-S,S=t.F[d]),n.$f(t,l.Oeb(it),bt);for($t=0;$t<2*t.I;$t++)n.$f(t,Ct(t.D[$t]),mt);for(O=0;O<t.R.g;O++)Qt=(n.gm(t.R,t.C[O])&Le)!=0||n.hm(t.R,t.C[O])==16?1:n.Xp(t.R,t.C[O])?0:j.Math.min(3,n.em(t.R,t.C[O])),n.$f(t,l.Oeb(Qt),2);if(A=0,(t.Q&z1)==0)for(m=0;m<t.R.f;m++)t._[t.B[m]]!=0&&t._[t.B[m]]!=3&&++A;if(n.$f(t,l.Oeb(A),mt),(t.Q&z1)==0)for(m=0;m<t.R.f;m++)t._[t.B[m]]!=0&&t._[t.B[m]]!=3&&(n.$f(t,l.Oeb(m),mt),t.bb[t.B[m]]==0?n.$f(t,Ct(t._[t.B[m]]),3):(Zt=t._[t.B[m]]==1?t.bb[t.B[m]]==1?4:6:t.bb[t.B[m]]==1?5:7,n.$f(t,l.Oeb(Zt),3),n.$f(t,Ct(t.ab[t.B[m]]),3)));if(e=0,(t.Q&z1)==0)for(K=0;K<t.R.g;K++)t.j[t.C[K]]!=0&&t.j[t.C[K]]!=3&&(!n.iq(t.R,t.C[K])||n.hm(t.R,t.C[K])==1)&&++e;if(n.$f(t,l.Oeb(e),mt),(t.Q&z1)==0)for(K=0;K<t.R.g;K++)t.j[t.C[K]]!=0&&t.j[t.C[K]]!=3&&(!n.iq(t.R,t.C[K])||n.hm(t.R,t.C[K])==1)&&(n.$f(t,l.Oeb(K),mt),n.hm(t.R,t.C[K])==1?t.n[t.C[K]]==0?n.$f(t,Ct(t.j[t.C[K]]),3):(Zt=t.j[t.C[K]]==1?t.n[t.C[K]]==1?4:6:t.n[t.C[K]]==1?5:7,n.$f(t,l.Oeb(Zt),3),n.$f(t,Ct(t.k[t.C[K]]),3)):n.$f(t,Ct(t.j[t.C[K]]),2));for(n.$f(t,Ct(t.R.K?1:0),1),rt=0,C=0;C<t.R.f;C++)n.Ll(t.R,t.B[C])!=0&&++rt;if(rt!=0)for(n.dg(t,1),n.$f(t,l.Oeb(rt),mt),m=0;m<t.R.f;m++)n.Ll(t.R,t.B[m])!=0&&(n.$f(t,l.Oeb(m),mt),n.$f(t,Ct(n.Ll(t.R,t.B[m])),8));if(t.w=0,t.R.K){for(n.ff(t,0,mt,z1,1,-1),n.ff(t,3,mt,Y1,1,-1),n.ff(t,4,mt,120,4,3),n.ff(t,5,mt,ZA,2,1),n.ff(t,6,mt,1,1,-1),n.ff(t,7,mt,Z0,4,7),rt=0,m=0;m<t.R.f;m++)n.Il(t.R,t.B[m])!=null&&++rt;if(rt>0){for(n.dg(t,8),n.$f(t,l.Oeb(rt),mt),v=0;v<t.R.f;v++)if(M=n.Il(t.R,t.B[v]),M!=null)for(n.$f(t,l.Oeb(v),mt),n.$f(t,Ct(M.length),4),r=M,i=0,D=r.length;i<D;++i)P=r[i],n.$f(t,l.Oeb(P),8);}n.gf(t,9,mt,384,2,7),n.gf(t,10,mt,31,5,0),n.ff(t,11,mt,Q0,1,-1),n.gf(t,12,mt,Le,8,9),n.ff(t,13,mt,$e,3,14),n.ff(t,14,mt,ae,5,17),n.ff(t,16,mt,Be,3,22);}for(rt=0,L=0;L<t.R.f;L++)t.a!=null&&t.a[t.B[L]]!=-1&&++rt;if(rt!=0)for(n.dg(t,17),n.$f(t,l.Oeb(rt),mt),m=0;m<t.R.f;m++)t.a!=null&&t.a[t.B[m]]!=-1&&(n.$f(t,l.Oeb(m),mt),n.$f(t,Ct(t.a[t.B[m]]),4));if((t.Q&8)!=0||(t.Q&i0)!=0){for(rt=0,Nt=0,m=0;m<t.R.f;m++)xt=n.Dl(t.R,t.B[m]),xt!=null&&(++rt,Nt=j.Math.max(Nt,w.Xzb(xt).length));if(rt!=0){for(Ot=n.Og(Nt),n.dg(t,18),n.$f(t,l.Oeb(rt),mt),n.$f(t,l.Oeb(Ot),4),v=0;v<t.R.f;v++)if(yt=n.Dl(t.R,t.B[v]),yt!=null)for(n.$f(t,l.Oeb(v),mt),n.$f(t,Ct(w.Xzb(yt).length),Ot),wt=0;wt<w.Xzb(yt).length;wt++)n.$f(t,(w.Szb(wt,w.Xzb(yt).length),Ct(w.Xzb(yt).charCodeAt(wt))),7);}}for(t.R.K&&(n.ff(t,19,mt,je,3,25),n.gf(t,20,mt,ie,3,17)),rt=0,s=0;s<t.R.f;s++)n.Ol(t.R,t.B[s])!=0&&++rt;if(rt!=0)for(n.dg(t,21),n.$f(t,l.Oeb(rt),mt),u=0;u<t.R.f;u++)n.Ol(t.R,t.B[u])!=0&&(n.$f(t,l.Oeb(u),mt),n.$f(t,Ct(n.Ol(t.R,t.B[u])>>4),2));if(t.R.K&&(n.ff(t,22,mt,Ie,1,-1),n.gf(t,23,mt,se,1,-1),n.gf(t,24,mt,oA,2,21)),(t.Q&16)!=0){for(u=0;u<t.R.f;u++)if(n.Om(t.R,t.B[u])){for(n.dg(t,25),P=0;P<t.R.f;P++)n.$f(t,Ct(n.Om(t.R,t.B[P])?1:0),1);break;}}if(vt=n.ig(t),vt!=null){for(rt=0,K=0;K<t.R.g;K++)vt[t.C[K]]&&++rt;for(n.dg(t,26),n.$f(t,l.Oeb(rt),mt),N=0;N<t.R.g;N++)vt[t.C[N]]&&n.$f(t,l.Oeb(N),mt);}for(t.R.K&&n.ff(t,27,mt,r0,1,-1),rt=0,I=0;I<t.R.g;I++)n.hm(t.R,t.C[I])==16&&++rt;if(rt!=0)for(n.dg(t,28),n.$f(t,l.Oeb(rt),mt),K=0;K<t.R.g;K++)n.hm(t.R,t.C[K])==16&&n.$f(t,l.Oeb(K),mt);for(t.R.K&&(n.ff(t,29,mt,he,2,30),n.ff(t,30,mt,Me,7,32),n.ff(t,32,mt,Dr,2,44),n.ff(t,33,mt,Ln,5,39),n.ff(t,34,mt,AA,1,-1),n.gf(t,35,mt,yn,1,-1),n.gf(t,36,mt,96,2,5)),rt=0,ut=0;ut<t.R.g;ut++)(n.hm(t.R,t.C[ut])==32||n.hm(t.R,t.C[ut])==64)&&++rt;if(rt!=0)for(n.dg(t,37),n.$f(t,l.Oeb(rt),mt),V=0;V<t.R.g;V++)(n.hm(t.R,t.C[V])==32||n.hm(t.R,t.C[V])==64)&&(n.$f(t,l.Oeb(V),mt),n.$f(t,Ct(n.hm(t.R,t.C[V])==32?0:1),1));if(t.b&&(t.Q&16)!=0){for(rt=0,Lt=0,m=0;m<t.R.f;m++){for(ot=0,Ht=n.Zo(t.R,t.B[m]);Ht<n.Ko(t.R,t.B[m]);Ht++)n.Om(t.R,n.Yo(t.R,t.B[m],Ht))&&(ct=Ht-n.Zo(t.R,t.B[m]),ot|=1<<ct,Lt=j.Math.max(Lt,ct+1));ot!=0&&++rt;}if(rt!=0)for(n.dg(t,38),n.$f(t,l.Oeb(rt),mt),n.$f(t,l.Oeb(Lt),3),u=0;u<t.R.f;u++){for(ot=0,Ht=n.Zo(t.R,t.B[u]);Ht<n.Ko(t.R,t.B[u]);Ht++)n.Om(t.R,n.Yo(t.R,t.B[u],Ht))&&(ct=Ht-n.Zo(t.R,t.B[u]),ot|=1<<ct);ot!=0&&(n.$f(t,l.Oeb(u),mt),n.$f(t,l.Oeb(ot),Lt));}}t.R.K&&n.ff(t,39,mt,ur,4,47),n.$f(t,0,1),t.J=n._f(t);},n.zg=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(t._=z(l._6,J,6,t.R.f,15,1),e=0;e<t.R.f;e++)if(t.db[e]==1||t.db[e]==2){if(o=t.eb[e],n.Vp(t.R,e))for(u=0;u<n.Zo(t.R,e);u++){for(r=n.Yo(t.R,e,u),s=0,L=z(l.d7,H,6,3,15,1),Q=0;Q<n.Zo(t.R,r);Q++)L[s]=n.Yo(t.R,r,Q),L[s]!=e&&++s;s==2&&t.f[L[0]]>t.f[L[1]]^t.H[L[0]]<t.H[L[1]]&&(o=!o);}else for(u=1;u<n.Zo(t.R,e);u++)for(Q=0;Q<u;Q++)i=n.Yo(t.R,e,u),D=n.Yo(t.R,e,Q),t.f[i]>t.f[D]&&(o=!o),t.H[i]<t.H[D]&&(o=!o);t._[e]=t.db[e]==1^o?1:2;}else t._[e]=t.db[e];for(t.j=z(l._6,J,6,t.R.g,15,1),A=0;A<t.R.g;A++)if(t.o[A]==1||t.o[A]==2){for(o=false,u=0;u<2;u++)if(P=n.Yl(t.R,u,A),n.Zo(t.R,P)==3){for(L=z(l.d7,H,6,2,15,1),s=0,Q=0;Q<3;Q++)n.Yo(t.R,P,Q)!=n.Yl(t.R,1-u,A)&&(L[s++]=n.Yo(t.R,P,Q));t.f[L[0]]>t.f[L[1]]&&(o=!o),t.H[L[0]]<t.H[L[1]]&&(o=!o);}t.j[A]=t.o[A]==1^o?1:2;}else t.j[A]=t.o[A];},n.Ag=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c;for(o=z(l.d7,H,6,32,15,1),Q=0,P=0;P<t.R.f;P++)(t._[P]==1||t._[P]==2)&&t.bb[P]==e&&(u=t.ab[P],o[u]<t.f[P]&&(o[u]==0&&++Q,o[u]=t.f[P]));for(i=0;i<t.R.g;i++)(t.j[i]==1||t.j[i]==2)&&t.n[i]==e&&n.hm(t.R,i)==1&&(u=t.k[i],c=j.Math.max(t.f[n.Yl(t.R,0,i)],t.f[n.Yl(t.R,1,i)]),o[u]<c&&(o[u]==0&&++Q,o[u]=c));for(D=z(l._6,J,6,32,15,1),L=0;L<Q;L++){for(h=-1,b=0,s=0;s<32;s++)b<o[s]&&(b=o[s],h=s);o[h]=0,D[h]=L<<24>>24;}for(A=0;A<t.R.f;A++)(t._[A]==1||t._[A]==2)&&t.bb[A]==e&&(t.ab[A]=D[t.ab[A]]);for(r=0;r<t.R.g;r++)(t.j[r]==1||t.j[r]==2)&&t.n[r]==e&&n.hm(t.R,r)==1&&(t.k[r]=D[t.k[r]]);},n.Bg=function(t,e,A){var P,r;for(t.U=e,P=0;P<t.R.f;P++)t.f[P]=A[P],t.db[P]=0,t.ib[P]=false;for(r=0;r<t.R.g;r++)t.o[r]=0,t.q[r]=false;},n.Cg=function(t,e){var A,P,r;if(A=n.Yl(t.R,0,e),A>=t.R.f)return  false;if(t.db[A]==1||t.db[A]==2)return  true;if(t.db[A]==3)return  false;if(P=n.Go(t.R,A),P!=-1)return t.o[P]==1||t.o[P]==2;for(r=0;r<n.Zo(t.R,A);r++)if(n._o(t.R,A,r)==2&&(t.db[n.Yo(t.R,A,r)]==1||t.db[n.Yo(t.R,A,r)]==2))return  true;return  false;},n.Dg=function(t,e){var A;for(A=0;A<n.Zo(t.R,e);A++)if(t.o[n.$o(t.R,e,A)]==3&&n._o(t.R,e,A)==1)return  true;return  false;},n.Eg=function(t){var e,A,P,r,i,D,u;for(D=z(l.d7,H,6,t.U+1,15,1),A=0;A<t.R.f;A++)n.Gl(t.R,A)==0&&(t.db[A]==1?++D[t.f[A]]:t.db[A]==2&&--D[t.f[A]]);for(r=0;r<t.R.g;r++)n.em(t.R,r)==1&&n.bm(t.R,r)==0&&(t.o[r]==1?(++D[t.f[n.Yl(t.R,0,r)]],++D[t.f[n.Yl(t.R,1,r)]]):t.o[r]==2&&(--D[t.f[n.Yl(t.R,0,r)]],--D[t.f[n.Yl(t.R,1,r)]]));for(u=1;u<=t.U;u++)if(D[u]!=0){if(i=D[u]<0,i){for(e=0;e<t.R.f;e++)n.Gl(t.R,e)==0&&(t.db[e]==1?t.db[e]=2:t.db[e]==2&&(t.db[e]=1));for(P=0;P<t.R.g;P++)n.em(t.R,P)==1&&n.bm(t.R,P)==0&&(t.o[P]==1?t.o[P]=2:t.o[P]==2&&(t.o[P]=1));}return i;}return  false;},n.Fg=function(t,e){var A;return A=z(l.d7,H,6,e,15,1),T.Vjb(t,0,A,0,t.length),A;},n.Gg=function(t,e){var A;return A=z(l.zeb,ht,6,e,16,1),T.Vjb(t,0,A,0,t.length),A;},n.Hg=function(t){var e,A;if(t.$!=null)for(e=0;e<t.R.f;e++)n.gn(t.R,e,t.$[e]);if(t.i!=null)for(A=0;A<t.R.g;A++)n.En(t.R,A,t.i[A]);},n.Ig=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(e=0;e<t.R.f;e++)if(t.db[e]==1||t.db[e]==2){if(o=false,n.Vp(t.R,e))for(u=0;u<n.Zo(t.R,e);u++){for(r=n.Yo(t.R,e,u),s=0,L=z(l.d7,H,6,3,15,1),Q=0;Q<n.Zo(t.R,r);Q++)L[s]=n.Yo(t.R,r,Q),L[s]!=e&&++s;s==2&&t.f[L[0]]>t.f[L[1]]^L[0]<L[1]&&(o=!o);}else for(u=1;u<n.Zo(t.R,e);u++)for(Q=0;Q<u;Q++)i=n.Yo(t.R,e,u),D=n.Yo(t.R,e,Q),t.f[i]>t.f[D]&&(o=!o),i<D&&(o=!o);n.tn(t.R,e,t.db[e]==1^o?1:2,t.fb[e]);}else n.tn(t.R,e,t.db[e],t.fb[e]);for(A=0;A<t.R.g;A++)if(t.o[A]==1||t.o[A]==2){for(o=false,u=0;u<2;u++)if(P=n.Yl(t.R,u,A),n.Zo(t.R,P)==3){for(L=z(l.d7,H,6,2,15,1),s=0,Q=0;Q<3;Q++)n.Yo(t.R,P,Q)!=n.Yl(t.R,1-u,A)&&(L[s++]=n.Yo(t.R,P,Q));t.f[L[0]]>t.f[L[1]]&&(o=!o),L[0]<L[1]&&(o=!o);}n.In(t.R,A,t.o[A]==1^o?1:2,t.p[A]);}else n.In(t.R,A,t.o[A],t.p[A]);},n.Jg=function(t){var e;for(e=0;e<t.R.f;e++)n.xn(t.R,e,t.M[e]);},n.Kg=function(t){var e,A,P;for(e=0;e<t.R.f;e++)!n.vm(t.R,e)&&t.db[e]==3&&n.kn(t.R,e,true);for(A=0;A<t.R.g;A++)t.o[A]==3&&(P=n.em(t.R,A),P==1?n.kn(t.R,n.Yl(t.R,0,A),true):P==2&&n.Ln(t.R,A,386));},n.Lg=function(t,e){return t<e?t:t-e;},n.Mg=function(t){n.Ng.call(this,t,0);},n.Ng=function(t,e){this.R=t,this.Q=e,n.aw(this.R,7),this.c=n.Og(this.R.f),(this.Q&z1)==0&&n.Bf(this),this.jb=(e&64)!=0||n.um(this.R),this.b=n.mp(this.R)==0,(this.Q&z1)==0&&(this.db=z(l._6,J,6,this.R.f,15,1),this.fb=z(l.zeb,ht,6,this.R.f,16,1),this.ib=z(l.zeb,ht,6,this.R.f,16,1),this.o=z(l._6,J,6,this.R.g,15,1),this.q=z(l.zeb,ht,6,this.R.g,16,1),this.p=z(l.zeb,ht,6,this.R.g,16,1)),this.d=false,n.Ff(this),(this.Q&z1)==0&&n.Lf(this),n.Kf(this);},n.Og=function(t){var e;for(e=0;t>0;)t>>=1,++e;return e;},x(34,1,{},n.Mg,n.Ng),y.b=false,y.c=0,y.d=false,y.r=false,y.t=0,y.v=0,y.w=0,y.G=false,y.I=0,y.K=false,y.L=false,y.O=0,y.Q=0,y.T=0,y.U=0,y.Z=false,y.jb=false,l.v7=X(34),n.Pg=function(t,e){var A,P,r,i;if(t.d!=e.d)return t.d>e.d?1:-1;for(r=t.a.length,i=e.a.length,A=j.Math.min(r,i),P=0;P<A;P++)if(--r,--i,t.a[r]!=e.a[i])return t.a[r]>e.a[i]?1:-1;return r!=i?r>i?1:-1:t.b!=e.b?t.b>e.b?1:-1:0;},n.Qg=function(t,e){return n.Pg(t,e);},n.Rg=function(){},x(266,1,{},n.Rg),y.Qb=function(t,e){return n.Qg(t,e);},y.lb=function(t){return this===t;},l.m7=X(266),n.Ug=function(){},x(173,1,{173:1},n.Ug),y.b=0,y.c=0,y.d=0,l.l7=X(173),n.Vg=function(t,e){return t.c!=e.c?t.c>e.c?1:-1:0;},n.Wg=function(t,e){return n.Vg(t,e);},n.Xg=function(){},x(267,1,{},n.Xg),y.Qb=function(t,e){return n.Wg(t,e);},y.lb=function(t){return this===t;},l.o7=X(267),n.$g=function(){},x(174,1,{174:1},n.$g),y.a=0,y.b=0,y.c=0,l.n7=X(174),n._g=function(t,e,A){t.b==0&&(++t.c,t.b=63),t.b==63?(t.d[t.c]=Pt(t.d[t.c],A),t.b-=e):t.b>=e?(t.d[t.c]=b1(t.d[t.c],e),t.d[t.c]=Pt(t.d[t.c],A),t.b-=e):(t.d[t.c]=b1(t.d[t.c],t.b),t.d[t.c]=Pt(t.d[t.c],t0(A,e-t.b)),e-=t.b,++t.c,t.b=63-e,t.d[t.c]=Pt(t.d[t.c],G(A,l.Oeb((1<<e)-1))));},n.ah=function(t,e){t.d[t.c]=B1(t.d[t.c],e);},n.bh=function(t,e){var A;for(A=0;A<t.c;A++)if(kt(t.d[A],e.d[A]))return dn(t.d[A],e.d[A])?-1:1;return M1(t.d[t.c],e.d[t.c])?0:dn(t.d[t.c],e.d[t.c])?-1:1;},n.dh=function(t,e){t.a=e,t.c=0,t.b=63,U.Jqb(t.d);},n.eh=function(t){this.d=z(l.e7,n0,6,t,14,1);},x(101,1,{101:1,32:1},n.eh),y.Rb=function(t){return n.bh(this,t);},y.a=0,y.b=0,y.c=0,l.p7=X(101),n.gh=function(t,e){return t.b!=e.b?t.b>e.b?-1:1:t.c!=e.c?t.c>e.c?-1:1:0;},n.hh=function(t,e,A){this.b=j.Math.max(t,e),this.c=j.Math.min(t,e),this.a=A;},x(172,1,{172:1,32:1},n.hh),y.Rb=function(t){return n.gh(this,t);},y.a=0,y.b=0,y.c=0,l.q7=X(172),n.jh=function(t,e,A,P){this.a=(w.Hzb(e),U.Gqb(t,z(l.d7,H,6,e,15,1),0,e)),this.b=(w.Hzb(P),U.Gqb(A,z(l.d7,H,6,P,15,1),0,P));},x(208,1,{208:1},n.jh),l.r7=X(208),n.kh=function(t,e){var A,P;for(P=0;P<t.g[e].length;P++)if(A=t.g[e][P],t.f[A]&&(t.o[A]==1||t.o[A]==2)&&t.k[A]==0)return  true;return  false;},n.lh=function(t,e,A){var P,r,i,D,u;for(r=0,D=0,u=0;u<t.g[e].length;u++)P=t.g[e][u],t.k[P]==A&&(i=1<<t.j[P],(D&i)==0&&(D|=i,++r));return r;},n.mh=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h;for(L=new U.Swb(),s=0;s<t.i.f;s++)if(n.Po(t.i,s)<2||n.Zo(t.i,s)>2)for(D=1;D<n.Zo(t.i,s);D++)for(e=n.Yo(t.i,s,D),u=0;u<D;u++)A=n.Yo(t.i,s,u),n.rh(t,e,A)&&(o=n.yh(t,e,A),o&&n.Fr(o,t.f)&&U.Uvb(L.a,o,(T.Kgb(),kA))==null);for(h=0;h<t.i.g;h++)t.c[h]!=0&&(n.em(t.i,h)!=2||t.c[h]!=2)||(e=n.Yl(t.i,0,h),A=n.Yl(t.i,1,h),n.rh(t,e,A)&&(o=n.yh(t,e,A),o&&n.Fr(o,t.f)&&U.Uvb(L.a,o,(T.Kgb(),kA))==null));for(t.g=z(l.d7,S1,7,L.a.c,0,2),t.e=z(l.zeb,ht,6,t.i.f,16,1),i=0,Q=(r=new U.kwb(new U.qwb(new U.xpb(L.a).a).b),new U.Gpb(r));U.fob(Q.a.a);)for(o=(P=U.iwb(Q.a),P.fd()),t.g[i++]=o.b,D=0;D<o.b.length;D++)t.e[o.b[D]]=true;},n.nh=function(t,e,A,P){var r,i,D,u,o,Q,L;for(r=z(l.d7,H,6,n.Zo(t.i,A),15,1),u=0,D=0;D<n.Zo(t.i,A);D++)r[u]=n.Yo(t.i,A,D),!P[r[u]]&&n.rh(t,e,r[u])&&++u;if(u==0)return  -1;if(u==1)return r[0];for(o=-1,Q=St,i=0;i<u;i++)L=n.Ap(t.i,e,r[i],St,P),L<Q&&(Q=L,o=r[i]);return o;},n.oh=function(t,e){return t.f[e]&&(t.o[e]==1||t.o[e]==2);},n.ph=function(t,e){return t.e[e];},n.qh=function(t){var e,A;for(A=true,e=0;e<t.i.f;e++)if(t.o[e]!=0&&!t.e[e]){A=false;break;}return A;},n.rh=function(t,e,A){var P,r,i,D,u;if(e==A||t.a[e]!=t.a[A]||t.o[e]!=0&&(t.o[e]==3||t.o[A]==3||t.p[e]^t.o[e]!==t.o[A]||t.k[e]!==t.k[A]||t.j[e]!==t.j[A])||(P=n.Wo(t.i,e,A),P!=-1&&(n.em(t.i,P)==1&&t.c[P]!=0||n.em(t.i,P)==2&&t.c[P]==1)))return  false;if(n.Po(t.i,e)==1&&!n.Sp(t.i,e)){for(r=-1,u=0;u<n.Zo(t.i,e);u++)if(n.Yo(t.i,e,u)!=A&&n._o(t.i,e,u)==2){r=n.$o(t.i,e,u);break;}for(i=-1,D=0;D<n.Zo(t.i,A);D++)if(n.Yo(t.i,A,D)!=e&&n._o(t.i,A,D)==2){i=n.$o(t.i,A,D);break;}if(r!=-1&&t.c[r]!=0&&t.d[r]^t.c[r]===t.c[i])return  false;}return  true;},n.sh=function(t,e,A,P,r){var i,D,u,o,Q,L;for(o=null,i=null,L=0;L<t.g[e].length;L++)D=t.g[e][L],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?i=n.Ah(i,(r[D]<<16)+D):t.k[D]==P&&t.j[D]==A&&(o=n.Ah(o,(r[D]<<16)+D)));if(u=n.Oh(o,i),u==0)return  false;if(u<0)for(Q=0;Q<t.g[e].length;Q++)D=t.g[e][Q],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?(t.k[D]=P<<24>>24,t.j[D]=A<<24>>24):t.k[D]==P&&t.j[D]==A&&(t.k[D]=0,t.j[D]=-1));return  true;},n.th=function(t,e){var A,P,r,i,D,u;if(!t.b)return  false;for(r=false,i=t.b.a.length-1;i>=0;i--)if(P=false,D=U.wj(t.b,i),D.a==2?P=n.sh(t,D.b,D.c,D.d,e):D.a==1&&(P=n.xh(t,D.b,e)),P){for(U.Ej(t.b,D),u=0;u<t.g[D.b].length;u++)A=t.g[D.b][u],t.n[A]=false;r=true;}return r;},n.uh=function(t){var e,A,P,r,i,D,u,o;if(t.g!=null)for(D=new n.Nh(t),t.b=new U.Lj(),r=0;r<t.g.length;r++)P=n.Fh(D,r),P==0?(n.Bh(D,r),u=n.lh(t,r,2),e=n.lh(t,r,1),A=n.kh(t,r),u==1&&e==1&&!A&&(n.wh(t,r,D.a+D.f++),U.rj(t.b,new n.ik(r,1,-1,-1))),u>0?(A&&(n.vh(t,r,D.i+D.g++,2),++u),U.rj(t.b,new n.ik(r,1,-1,-1))):e>0?(A&&n.vh(t,r,D.a+D.f++,1),U.rj(t.b,new n.ik(r,1,-1,-1))):A&&(n.vh(t,r,D.a+D.f++,1),U.rj(t.b,new n.ik(r,1,-1,-1)))):P==1&&(n.kh(t,r)?(i=n.Eh(D,r),o=n.Gh(D,r),U.rj(t.b,new n.ik(r,2,i,o))):(n.Bh(D,r),U.rj(t.b,new n.ik(r,1,-1,-1))));},n.vh=function(t,e,A,P){var r,i;for(i=0;i<t.g[e].length;i++)r=t.g[e][i],t.f[r]&&(t.o[r]==1||t.o[r]==2)&&t.k[r]==0&&(t.k[r]=P<<24>>24,t.j[r]=A<<24>>24);},n.wh=function(t,e,A){var P,r;for(r=0;r<t.g[e].length;r++)P=t.g[e][r],t.k[P]==2&&(t.k[P]=1,t.j[P]=A<<24>>24);},n.xh=function(t,e,A){var P,r,i,D,u,o,Q,L;for(i=t.g[e],r=1,o=0;o<i.length;o++)if(P=i[o],t.f[P]&&t.k[P]==2){r=2;break;}for(D=z(l.d7,S1,7,32,0,2),Q=0;Q<i.length;Q++)P=i[Q],t.f[P]&&t.k[P]==r&&(D[t.j[P]]=n.Ah(D[t.j[P]],(A[P]<<16)+P));for(L=0;L<32;L++)D[L]!=null&&w.qzb(D[L],w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[])));if(U.$qb(D,new n.Ph()),n.Oh(D[0],D[1])==0)return  false;for(u=0;u<D[0].length;u++)P=D[0][u]&r1,t.k[P]=0,t.j[P]=-1;return  true;},n.yh=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;for(o=z(l.d7,H,6,t.i.f,15,1),m=z(l.d7,H,6,t.i.f,15,1),c=z(l.zeb,ht,6,t.i.f,16,1),Q=z(l.zeb,ht,6,t.i.f,16,1),C=new n.Gr(t.i.f),o[0]=e,m[e]=A,m[A]=-2,C.a[e]=true,C.a[A]=true,i=0,L=0;i<=L;){if(D=o[i],m[D]==D){for(s=0;s<n.Zo(t.i,D);s++)if(P=n.Yo(t.i,D,s),!C.a[P]){if(n._o(t.i,D,s)==2&&n.Sl(t.i,P)<10)o[++L]=P,m[P]=P,Q[P]=Q[D]||n.Po(t.i,P)==2,c[P]=Q[D]&&!c[D],C.a[P]=true;else if(Q[D]&&c[D]){if(v=n.nh(t,P,m[D],C.a),v==-1)return null;o[++L]=P,m[P]=v,m[v]=-2,Q[P]=false,C.a[P]=true,C.a[v]=true;}else if(n.fq(t.i,n.$o(t.i,D,s))&&(o[++L]=P,m[P]=P,Q[P]=false,C.a[P]=true,(n.Sl(t.i,P)==6&&n.Po(t.i,P)==0||n.Sl(t.i,P)==7&&n.Al(t.i,P)==1||n.Sl(t.i,P)==14||n.Sl(t.i,P)==15&&n.Zo(t.i,P)>2||n.Sl(t.i,P)==16&&n.Zo(t.i,P)>2)&&n.Zo(t.i,P)>2)){for(u=false,g=1;g<n.Zo(t.i,P);g++)if(M=n.Yo(t.i,P,g),!C.a[M])for(d=0;d<g;d++)S=n.Yo(t.i,P,d),C.a[S]||n.rh(t,M,S)&&(o[++L]=M,m[M]=S,m[S]=-2,Q[M]=false,C.a[M]=true,C.a[S]=true,u=true);if(!u)return null;}}}else {for(r=z(l.zeb,ht,6,n.Zo(t.i,D),16,1),h=0;h<n.Zo(t.i,D);h++)if(P=n.Yo(t.i,D,h),C.a[P])r[h]=m[P]==P;else for(g=0;g<n.Zo(t.i,P);g++)if(n.Yo(t.i,P,g)==m[D]){r[h]=true;break;}for(b=0;b<n.Zo(t.i,D);b++)if(r[b])if(P=n.Yo(t.i,D,b),C.a[P]){if(n.Wo(t.i,P,m[D])==-1)return null;}else o[++L]=P,m[P]=P,c[P]=false,Q[P]=true,C.a[P]=true;for(s=0;s<n.Zo(t.i,D);s++)if(!r[s]&&(P=n.Yo(t.i,D,s),!C.a[P])){if(v=n.nh(t,P,m[D],C.a),v==-1)return null;o[++L]=P,m[P]=v,m[v]=-2,Q[P]=false,C.a[P]=true,C.a[v]=true;}}++i;}return C;},n.zh=function(t,e,A,P,r,i,D,u,o,Q){this.i=t,this.a=e,this.f=A,this.o=P,this.c=r,this.k=i,this.j=D,this.p=u,this.d=o,this.n=Q,n.mh(this);},n.Ah=function(t,e){var A,P;for(P=z(l.d7,H,6,t==null?1:t.length+1,15,1),A=0;A<P.length-1;A++)P[A]=t[A];return P[P.length-1]=e,P;},x(265,1,{},n.zh),l.t7=X(265),n.Bh=function(t,e){var A,P,r,i,D,u;for(r=0;r<t.b;r++)if(t.e[r][e]&&t.c[r]!=-3){for(P=0;P<=t.j.g.length;P++)if(P!=e&&t.e[r][P])for(t.e[r][e]=false,u=r<t.a?r:r<t.b?r-t.a:-1,D=n.Lh(t,r<t.a?1:r<t.b?2:0),i=0;i<t.j.g[e].length;i++)A=t.j.g[e][i],n.oh(t.j,A)&&t.j.j[A]==u&&(t.j.j[A]=D<<24>>24);}},n.Ch=function(t,e,A){var P,r,i,D,u;for(u=false,D=1,e[A]=D,P=true;P;){for(P=false,r=0;r<t.b;r++)if(e[r]==D)for(i=0;i<t.b;i++)e[i]==0&&n.Kh(t,r,i)&&(t.c[i]==-2?(e[i]=D+1,P=true):t.c[i]!=t.c[A]&&(e[i]=D+1,u=true));++D;}return u;},n.Dh=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c;for(o=P+1;o<t.j.g.length;o++)if(o!=P&&t.e[e][o]&&t.e[A][o])return D=z(l.d7,H,6,2,15,1),D[0]=A,D[1]=e,D;for(c=z(l.d7,H,6,t.b,15,1),L=z(l.d7,H,6,t.b,15,1),Q=z(l.d7,H,6,t.b,15,1),i=0,s=0,Q[0]=e,L[e]=1;i<=s;){for(h=0;h<t.d[Q[i]].length;h++){if(r=t.d[Q[i]][h],r==A){if(i==0)continue;for(u=L[Q[i]]+1,D=z(l.d7,H,6,u,15,1),D[0]=r,D[1]=Q[i],b=2;b<u;b++)D[b]=c[D[b-1]];return D;}L[r]==0&&t.c[r]!=-3&&(L[r]=L[Q[i]]+1,Q[++s]=r,c[r]=Q[i]);}++i;}return null;},n.Eh=function(t,e){var A;for(A=0;A<t.b;A++)if(t.e[A][e]&&t.c[A]==-3)return A<t.a?A:A<t.b?A-t.a:-1;return  -1;},n.Fh=function(t,e){var A,P;for(A=0,P=0;P<t.b;P++)t.e[P][e]&&t.c[P]==-3&&++A;return A;},n.Gh=function(t,e){var A;for(A=0;A<t.b;A++)if(t.e[A][e]&&t.c[A]==-3)return A<t.a?1:A<t.b?2:0;return  -1;},n.Hh=function(t,e){return e<t.a?e:e<t.b?e-t.a:-1;},n.Ih=function(t,e){return e<t.a?1:e<t.b?2:0;},n.Jh=function(t,e){var A,P;return P=t.j.k[e],A=t.j.j[e],P==0?t.b:P==1?A:t.a+A;},n.Kh=function(t,e,A){var P;for(P=0;P<t.j.g.length;P++)if(t.e[e][P]&&t.e[A][P])return  true;return  false;},n.Lh=function(t,e){return e==1?t.a+t.f++:t.i+t.g++;},n.Mh=function(t,e){var A,P,r,i,D,u,o,Q,L,s;for(L=St,o=-1,s=-1,Q=-1,P=0;P<t.j.i.f;P++)if(n.oh(t.j,P)&&t.j.k[P]!=0)for(u=0;u<e.length;u++)r=n.Hh(t,e[u]),i=n.Ih(t,e[u]),t.j.k[P]==i&&t.j.j[P]==r&&L>t.j.a[P]+(i==1?q1:0)&&(L=t.j.a[P]+(i==1?q1:0),o=r,s=i,Q=e[u]);for(A=0;A<t.j.i.f;A++)n.oh(t.j,A)&&t.j.k[A]==s&&t.j.j[A]==o&&(t.j.k[A]=0,t.j.j[A]=-1);for(D=0;D<t.j.g.length;D++)t.e[Q][D]=false;},n.Nh=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(this.j=t,P=0;P<t.i.f;P++)t.f[P]&&(t.o[P]==1||t.o[P]==2)&&(t.k[P]==1?this.a<=t.j[P]&&(this.a=1+t.j[P]):t.k[P]==2&&this.i<=t.j[P]&&(this.i=1+t.j[P]));for(this.b=this.a+this.i,this.e=h1(l.zeb,[l1,ht],[12,6],16,[this.b+1,t.g.length+1],2),r=0;r<t.i.f;r++)t.f[r]&&(t.o[r]==1||t.o[r]==2)&&!t.e[r]&&(this.e[n.Jh(this,r)][t.g.length]=true);for(o=0;o<t.g.length;o++)for(d=0;d<t.g[o].length;d++)A=t.g[o][d],t.f[A]&&(t.o[A]==1||t.o[A]==2)&&(this.e[n.Jh(this,A)][o]=true);for(this.d=z(l.d7,S1,7,this.b,0,2),Q=0;Q<t.g.length;Q++)for(b=1;b<this.b;b++)if(this.e[b][Q])for(c=0;c<b;c++)this.e[c][Q]&&(this.d[b]=n.Ah(this.d[b],c),this.d[c]=n.Ah(this.d[c],b));for(this.c=z(l.d7,H,6,this.b+1,15,1),h=0;h<this.b;h++)this.e[h][t.g.length]?this.c[h]=-1:this.c[h]=-2;for(L=0;L<t.g.length;L++)if(this.e[this.b][L])for(s=0;s<this.b;s++)this.e[s][L]&&this.c[s]!=L&&(this.c[s]==-2?this.c[s]=L:this.c[s]=-3);for(e=0;e<this.b;e++)if(this.c[e]>=-1&&(i=z(l.d7,H,6,this.b,15,1),n.Ch(this,i,e)))for(s=0;s<this.b;s++)i[s]!=0&&(this.c[s]=-3);for(u=0;u<t.g.length-1;u++)for(b=1;b<this.b;b++)if(this.e[b][u]&&this.c[b]!=-3){for(c=0;c<b;c++)if(this.e[c][u]&&this.c[c]!=-3&&(D=n.Dh(this,b,c,u),D!=null)){for(g=0;g<D.length;g++)this.c[D[g]]=-3;n.Mh(this,D);break;}}},x(359,1,{},n.Nh),y.a=0,y.b=0,y.f=0,y.g=0,y.i=0,l.s7=X(359),n.Oh=function(t,e){var A,P;if(t==null)return e==null?0:1;if(e==null)return  -1;for(A=j.Math.min(t.length,e.length),P=0;P<A;P++)if((t[P]&fe)!=(e[P]&fe))return (t[P]&fe)<(e[P]&fe)?-1:1;return t.length==e.length?0:t.length<e.length?-1:1;},n.Ph=function(){},x(360,1,{},n.Ph),y.Qb=function(t,e){return n.Oh(t,e);},y.lb=function(t){return this===t;},l.u7=X(360),n.Sh=function(t,e){switch(e){case 0:return n.Th(t,false);case 1:return n.Vh(t);case 2:return n.Uh(t);case 3:return n.Xh(t);case 4:return n.Wh(t);}return null;},n.Th=function(t,e){try{return e?(t=n.ew(t),n.yq(t,!0),n.Sr(t),n.qg(new n.Mg(t))):n.qg(new n.Mg(t));}catch(A){if(A=H1(A),Jt(A,27))return T.Ujb(),null;throw Ut(A);}},n.Uh=function(t){var e,A;try{for(t=n.ew(t),n.ow(t),e=t.r,A=0;A<e;A++)t.H[A]=1,t.T=0;return n.qg(new n.Mg(t));}catch(P){if(P=H1(P),Jt(P,27))return T.Ujb(),null;throw Ut(P);}},n.Vh=function(t){try{return t=n.ew(t),n.ow(t),n.qg(new n.Mg(t));}catch(e){if(e=H1(e),Jt(e,27))return T.Ujb(),null;throw Ut(e);}},n.Wh=function(t){var e;try{return t=n.ew(t),n.ow(t),e=n.Gw(new n.Jw(t)),n.qg(new n.Ng(e,8));}catch(A){if(A=H1(A),Jt(A,27))return T.Ujb(),null;throw Ut(A);}},n.Xh=function(t){var e;try{return e=n.Gw(new n.Jw(t)),n.qg(new n.Ng(e,8));}catch(A){if(A=H1(A),Jt(A,27))return T.Ujb(),null;throw Ut(A);}},n.Yh=function(t,e,A,P){t.a+=e,t.b+=A,t.c+=P;},n.Zh=function(t,e){return t.a+=e.a,t.b+=e.b,t.c+=e.c,t;},n.$h=function(t,e){return new n.ni(t.a+e.a,t.b+e.b,t.c+e.c);},n._h=function(t,e){return t.a=(t.a+e.a)/2,t.b=(t.b+e.b)/2,t.c=(t.c+e.c)/2,t;},n.ai=function(t,e){return t.a!=e.a?t.a<e.a?-1:1:t.b!=e.b?t.b<e.b?-1:1:t.c!=e.c?t.c<e.c?-1:1:0;},n.bi=function(t,e){return new n.ni(t.b*e.c-t.c*e.b,-(t.a*e.c-t.c*e.a),t.a*e.b-t.b*e.a);},n.ci=function(t,e){return j.Math.sqrt((e.a-t.a)*(e.a-t.a)+(e.b-t.b)*(e.b-t.b)+(e.c-t.c)*(e.c-t.c));},n.di=function(t,e){var A;return e==null||!Jt(e,29)?false:(A=e,j.Math.abs(A.a-t.a)+j.Math.abs(A.b-t.b)+j.Math.abs(A.c-t.c)<1e-6);},n.ei=function(t,e){var A,P,r;return P=t.a*t.a+t.b*t.b+t.c*t.c,r=e.a*e.a+e.b*e.b+e.c*e.c,P==0||r==0||(A=(t.a*e.a+t.b*e.b+t.c*e.c)/j.Math.sqrt(P*r),A>=1)?0:A<=-1?n1:j.Math.acos(A);},n.fi=function(t,e){var A,P,r;return A=t.a,P=t.b,r=t.c,t.a=A*e[0][0]+P*e[1][0]+r*e[2][0],t.b=A*e[0][1]+P*e[1][1]+r*e[2][1],t.c=A*e[0][2]+P*e[1][2]+r*e[2][2],t;},n.gi=function(t,e){return t.a*=e,t.b*=e,t.c*=e,t;},n.hi=function(t,e,A,P){t.a=e,t.b=A,t.c=P;},n.ii=function(t,e){return n.hi(t,e.a,e.b,e.c),t;},n.ji=function(t,e){return t.a-=e.a,t.b-=e.b,t.c-=e.c,t;},n.ki=function(t,e){return new n.ni(t.a-e.a,t.b-e.b,t.c-e.c);},n.li=function(t){var e;return e=j.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),e==0?(T.Ujb(),t.a=1,t.b=0,t.c=0,t):(t.a/=e,t.b/=e,t.c/=e,t);},n.mi=function(){},n.ni=function(t,e,A){this.a=t,this.b=e,this.c=A;},n.oi=function(t){n.ni.call(this,t.a,t.b,t.c);},x(29,1,{29:1,4:1,32:1},n.mi,n.ni,n.oi),y.Rb=function(t){return n.ai(this,t);},y.lb=function(t){return n.di(this,t);},y.ob=function(){var t;return t=new o0.Mmb("0.00"),"["+o0.Imb(t,this.a)+", "+o0.Imb(t,this.b)+", "+o0.Imb(t,this.c)+"]";},y.a=0,y.b=0,y.c=0,l.w7=X(29),n.si=function(t,e){e.Mb(t.c),n.le(e,t.a,t.b);},n.ti=function(t,e){e.c*=t.c,e.a=e.a*t.c+t.a,e.b=e.b*t.c+t.b;},n.ui=function(t,e){n.cn(e,t.c),n.$n(e,t.a,t.b);},n.vi=function(t,e){e.a=e.a*t.c+t.a,e.b=e.b*t.c+t.b;},n.wi=function(t,e){e.c=e.c*t.c+t.a,e.d=e.d*t.c+t.b,e.b*=t.c,e.a*=t.c;},n.xi=function(t){t.a=0,t.b=0,t.c=1;},n.yi=function(t){return t.c==1&&t.a==0&&t.b==0;},n.zi=function(t,e){return e*t.c+t.a;},n.Ai=function(t,e){return e*t.c+t.b;},n.Bi=function(){n.xi(this);},n.Ci=function(t,e){this.c=1,this.a=t,this.b=e;},n.Di=function(t,e,A,P){var r,i,D,u;n.xi(this),e?(P&JA)==0?Tt.AN(e,t.c,t.d)&&Tt.AN(e,t.c+t.b,t.d+t.a)||((t.b>e.b||t.a>e.a)&&(i=e.b/t.b,u=e.a/t.a,this.c=j.Math.min(i,u)),t.c*this.c<e.c?this.a=e.c-t.c*this.c:(t.c+t.b)*this.c>e.c+e.b&&(this.a=e.c+e.b-(t.c+t.b)*this.c),t.d*this.c<e.d?this.b=e.d-t.d*this.c:(t.d+t.a)*this.c>e.d+e.a&&(this.b=e.d+e.a-(t.d+t.a)*this.c)):(i=e.b/t.b,u=e.a/t.a,D=P&r1,D==0?D=24:(P&A0)!=0&&(D/=256),r=D/A,this.c=j.Math.min(r,j.Math.min(i,u)),this.a=e.c+e.b/2-this.c*(t.c+t.b/2),this.b=e.d+e.a/2-this.c*(t.d+t.a/2)):(P&q1)!=0&&(D=(P&r1)!=0?P&r1:24,this.c=D/A);},x(78,1,{},n.Bi,n.Ci,n.Di),y.ob=function(){return "DepictorTransformation Offset: "+this.a+","+this.b+" Scaling: "+this.c;},y.a=0,y.b=0,y.c=0,l.x7=X(78),n.Fi=function(t){var e,A,P,r;return !T.ejb(w.Xzb(t).substr(0,21),ir)||(P=w.Xzb(YA).length,!T.ejb(w.Xzb(t).substr(w.Xzb(t).length-P,P),YA))||(A=T.mjb(t,Z1(34),21),A==-1)?null:(r=(w.Rzb(21,A,w.Xzb(t).length),w.Xzb(t).substr(21,A-21)),e=T.vjb(t,21+w.Xzb(r).length+1,w.Xzb(t).length-17),T.ejb(r,"arrow")?new tt.GH(e):T.ejb(r,"text")?new n.ax(e):null);},U.Drb=function(t){return new U.Fvb(t);},U.Erb=function(t){return new gt.$yb(t.Tb());},U.Gi=function(t,e){var A,P,r;for(w.Lzb(e),A=false,r=e.Sb();r.cd();)P=r.dd(),A=A|t.add(P);return A;},U.Hi=function(t,e,A){var P,r;for(r=t.Sb();r.cd();)if(P=r.dd(),k1(e)===k1(P)||e!=null&&T.Oc(e,P))return A&&r.ed(),true;return  false;},U.Ii=function(t,e){var A,P;for(w.Lzb(e),P=e.Sb();P.cd();)if(A=P.dd(),!t.contains(A))return  false;return  true;},U.Ji=function(t,e){var A,P,r;for(w.Lzb(e),A=false,P=t.Sb();P.cd();)r=P.dd(),e.contains(r)&&(P.ed(),A=true);return A;},U.Ki=function(t,e){var A,P,r;for(w.Lzb(e),A=false,P=t.Sb();P.cd();)r=P.dd(),e.contains(r)||(P.ed(),A=true);return A;},U.Li=function(t,e){var A,P,r,i;for(i=t.size(),e.length<i&&(e=w.rzb(new Array(i),e)),r=e,P=t.Sb(),A=0;A<i;++A)a1(r,A,P.dd());return e.length>i&&l.O5(e,i,null),e;},U.Mi=function(t){var e,A,P;for(P=new U.Kvb(", ","[","]"),A=t.Sb();A.cd();)e=A.dd(),U.Hvb(P,k1(e)===k1(t)?"(this Collection)":e==null?re:Zn(e));return P.a?w.Xzb(P.e).length==0?P.a.a:P.a.a+(""+P.e):P.c;},x(43,1,_A),y.Tb=function(){return new U.Fvb(this);},y.Ub=function(){return new gt.$yb(this.Tb());},y.add=function(t){throw Ut(new T.Yjb("Add not supported on this collection"));},y.addAll=function(t){return U.Gi(this,t);},y.clear=function(){var t;for(t=this.Sb();t.cd();)t.dd(),t.ed();},y.contains=function(t){return U.Hi(this,t,false);},y.containsAll=function(t){return U.Ii(this,t);},y.isEmpty=function(){return this.size()==0;},y.remove=function(t){return U.Hi(this,t,true);},y.removeAll=function(t){return U.Ji(this,t);},y.retainAll=function(t){return U.Ki(this,t);},y.toArray=function(){return this.Vb(z(l.Mbb,C1,1,this.size(),5,1));},y.Vb=function(t){return U.Li(this,t);},y.ob=function(){return U.Mi(this);},l.acb=X(43),U.xub=function(t){return new U.Fvb(t);},U.yub=function(t){var e;return gt.Yyb(t.Ub(),(e=gt.gyb(new gt.nyb(),gt.Wxb(new gt.lyb(),new gt.jyb(),new gt.Nyb(),F(k(l.sdb,1),s0,96,0,[(gt.$xb(),gt.Yxb)]))),gt.cyb(e,new gt.qyb())));},U._i=function(t,e){var A,P;for(A=0,P=t.size();A<P;++A)if(U.Mub(e,t.getAtIndex(A)))return A;return  -1;},U.aj=function(t,e,A){return new U.qob(t,e,A);},x(95,43,X8),y.Tb=function(){return new U.Fvb(this);},y.addAtIndex=function(t,e){throw Ut(new T.Yjb("Add not supported on this list"));},y.add=function(t){return this.addAtIndex(this.size(),t),true;},y.addAllAtIndex=function(t,e){var A,P,r;for(w.Lzb(e),A=false,r=e.Sb();r.cd();)P=r.dd(),this.addAtIndex(t++,P),A=true;return A;},y.clear=function(){this.Wb(0,this.size());},y.lb=function(t){var e,A,P,r,i;if(t===this)return  true;if(!Jt(t,83)||(i=t,this.size()!=i.size()))return  false;for(r=i.Sb(),A=this.Sb();A.cd();)if(e=A.dd(),P=r.dd(),!(k1(e)===k1(P)||e!=null&&T.Oc(e,P)))return  false;return  true;},y.nb=function(){return U.Grb(this);},y.indexOf=function(t){return U._i(this,t);},y.Sb=function(){return new U.iob(this);},y.lastIndexOf=function(t){var e;for(e=this.size()-1;e>-1;--e)if(U.Mub(t,this.getAtIndex(e)))return e;return  -1;},y.removeAtIndex=function(t){throw Ut(new T.Yjb("Remove not supported on this list"));},y.Wb=function(t,e){var A,P;for(P=new U.oob(this,t),A=t;A<e;++A)w.Jzb(P.a<P.c.size()),P.c.getAtIndex(P.b=P.a++),U.hob(P);},y.setAtIndex=function(t,e){throw Ut(new T.Yjb("Set not supported on this list"));},y.subList=function(t,e){return U.aj(this,t,e);},l.hcb=X(95);function Ju(f){return f==null?false:f.$implements__java_lang_Cloneable||Array.isArray(f);}U.pj=function(t){t.a=z(l.Mbb,C1,1,0,5,1);},U.qj=function(t,e,A){w.Nzb(e,t.a.length),w.kzb(t.a,e,A);},U.rj=function(t,e){return w.nzb(t.a,e),true;},U.sj=function(t,e,A){var P,r;return w.Nzb(e,t.a.length),P=A.toArray(),r=P.length,r==0?false:(w.lzb(t.a,e,P),true);},U.tj=function(t,e){var A,P;return A=e.toArray(),P=A.length,P==0?false:(w.lzb(t.a,t.a.length,A),true);},U.uj=function(t){w.pzb(t.a,0);},U.vj=function(t,e){return U.yj(t,e,0)!=-1;},U.wj=function(t,e){return w.Kzb(e,t.a.length),t.a[e];},U.xj=function(t,e){return U.yj(t,e,0);},U.yj=function(t,e,A){for(;A<t.a.length;++A)if(U.Mub(e,t.a[A]))return A;return  -1;},U.zj=function(t){return t.a.length==0;},U.Aj=function(t){return new U.wqb(t);},U.Bj=function(t,e){return U.Cj(t,e,t.a.length-1);},U.Cj=function(t,e,A){for(;A>=0;--A)if(U.Mub(e,t.a[A]))return A;return  -1;},U.Dj=function(t,e){var A;return A=(w.Kzb(e,t.a.length),t.a[e]),w.ozb(t.a,e,1),A;},U.Ej=function(t,e){var A;return A=U.yj(t,e,0),A==-1?false:(U.Dj(t,A),true);},U.Fj=function(t,e,A){var P;w.Ozb(e,A,t.a.length),P=A-e,w.ozb(t.a,e,P);},U.Gj=function(t,e,A){var P;return P=(w.Kzb(e,t.a.length),t.a[e]),t.a[e]=A,P;},U.Hj=function(t){return t.a.length;},U.Ij=function(t,e){U.Zqb(t.a,t.a.length,e);},U.Jj=function(t){return w.hzb(t.a);},U.Kj=function(t,e){var A,P;for(P=t.a.length,e.length<P&&(e=w.rzb(new Array(P),e)),A=0;A<P;++A)a1(e,A,t.a[A]);return e.length>P&&l.O5(e,P,null),e;},U.Lj=function(){U.pj(this);},U.Mj=function(t){U.pj(this),w.Fzb(t>=0,"Initial capacity must not be negative");},x(20,95,WA,U.Lj,U.Mj),y.addAtIndex=function(t,e){U.qj(this,t,e);},y.add=function(t){return U.rj(this,t);},y.addAllAtIndex=function(t,e){return U.sj(this,t,e);},y.addAll=function(t){return U.tj(this,t);},y.clear=function(){U.uj(this);},y.contains=function(t){return U.vj(this,t);},y.getAtIndex=function(t){return U.wj(this,t);},y.indexOf=function(t){return U.xj(this,t);},y.isEmpty=function(){return U.zj(this);},y.Sb=function(){return U.Aj(this);},y.lastIndexOf=function(t){return U.Bj(this,t);},y.removeAtIndex=function(t){return U.Dj(this,t);},y.remove=function(t){return U.Ej(this,t);},y.Wb=function(t,e){U.Fj(this,t,e);},y.setAtIndex=function(t,e){return U.Gj(this,t,e);},y.size=function(){return U.Hj(this);},y.toArray=function(){return U.Jj(this);},y.Vb=function(t){return U.Kj(this,t);},l.zcb=X(20),n.dk=function(t){var e,A,P;for(P=new T.Pjb(),A=new U.wqb(t);A.a<A.c.a.length;)e=U.vqb(A),T.Njb(T.Njb(P,ir+e.Kb()+'"'+e.Jb()+YA),p1);return P.a;},n.ek=function(){U.Lj.call(this);},n.fk=function(t){var e,A;U.Lj.call(this);try{if(t)for(A=new U.wqb(t);A.a<A.c.a.length;)e=U.vqb(A),U.rj(this,e.Fb());}catch(P){if(P=H1(P),!Jt(P,19))throw Ut(P);}},n.gk=function(t){var e,A,P;if(U.Lj.call(this),!(t==null||w.Xzb(t).length==0))for(e=0,A=T.ljb(t,Z1(10));A!=-1;)P=n.Fi((w.Rzb(e,A,w.Xzb(t).length),w.Xzb(t).substr(e,A-e))),P&&w.nzb(this.a,P),e=A+1,A=T.mjb(t,Z1(10),e);},x(103,20,WA,n.ek,n.fk,n.gk),y.ob=function(){return n.dk(this);},l.y7=X(103),n.ik=function(t,e,A,P){this.b=t,this.a=e,this.c=A,this.d=P;},x(109,1,{109:1},n.ik),y.a=0,y.b=0,y.c=0,y.d=0,l.z7=X(109),n.jk=function(t){var e,A,P,r,i,D;if(t.j!=0)return t.j;if(t.i&&n.Sl(t.e,t.a)!=15&&n.Sl(t.e,t.a)!=16){for(D=0;D<n.Ko(t.e,t.a);D++)if(i=n.$o(t.e,t.a,D),n.Rm(t.e,i,t.a))return n.Yo(t.e,t.a,D)==t.b?t.j=n.hm(t.e,i)==257?3:1:t.j=n.hm(t.e,i)==257?1:3,t.j;}return e=n.Xl(t.e,t.a,t.g),P=n.Xl(t.e,t.a,t.b),P<e&&(P+=X1),n.Ko(t.e,t.a)==2?(A=P-e,A>3.0915926535897933&&A<3.191592653589793?(t.j=-1,t.j):(t.j=A<n1?4:2,t.j)):(r=n.Xl(t.e,t.a,t.d),r<e&&(r+=X1),t.j=r<P?2:4,t.j);},n.kk=function(t,e,A,P){var r,i,D,u;for(this.e=t,this.g=A,this.a=P,D=-1,u=0;u<n.Ko(this.e,this.a);u++){if(r=n.Yo(this.e,this.a,u),i=n.$o(this.e,this.a,u),r==this.g){n.hm(this.e,i)==386&&(this.j=-1);continue;}n.Rm(this.e,i,this.a)&&(this.i&&(t.u[P]|=q1),this.i=true),D==e[r]?(this.d=r,this.f=true,this.c=n.fq(this.e,i)):D<e[r]?(D=e[r],this.d=this.b,this.b=r):this.d=r;}},x(117,1,{},n.kk),y.a=0,y.b=0,y.c=false,y.d=0,y.f=false,y.g=0,y.i=false,y.j=0,l.A7=X(117),n.lk=function(t,e,A,P){this.c=t,this.b=e,this.d=A,this.a=P;},x(18,1,{18:1},n.lk),y.ob=function(){return this.d;},y.a=0,y.c=0,l.B7=X(18),n.nk=function(t){var e,A,P,r,i;if(r=new U.Zub(new n.Gk()),i=new U.Wub(),t.k!=null)for(P=0;P<t.k.length;P++)for(e=0;e<t.k[P].r;e++)U.Sub(r,w.Xzb(t.e[P].R.c*n.cm(t.k[P],e))),U.Sub(i,U.Tub(r)),r.b.a.length<i.b.a.length&&U.Sub(r,U.Tub(i));return A=r.b.a.length+i.b.a.length,A==0?n.ok(t):t.p.c*(A%2==0?(T.Vhb(r.b.a.length==0?null:U.wj(r.b,0))+T.Vhb(i.b.a.length==0?null:U.wj(i.b,0)))/2:T.Vhb(r.b.a.length==0?null:U.wj(r.b,0)));},n.ok=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g;if(s=L0,h=or,c=L0,g=or,r=0,t.k!=null)for(u=0;u<t.k.length;u++)for(A=0;A<t.k[u].q;A++)L=n.zi(t.e[u].R,n.Cl(t.k[u],A).a),b=n.Ai(t.e[u].R,n.Cl(t.k[u],A).b),s=j.Math.min(s,L),h=j.Math.max(h,L),c=j.Math.min(c,b),g=j.Math.max(g,b),++r;return r<=1?t.d:(i=h-s,D=g-c,o=(i+D)/2,Q=3*r,e=i*D,P=o/(1-Q),j.Math.sqrt(P*P-e/(1-Q))-P);},n.pk=function(t){var e;if(t.p=new n.Bi(),t.k!=null)for(t.e=z(l.W9,C1,105,t.k.length,0,1),e=0;e<t.k.length;e++)t.e[e]=new Tt.YM(t.k[e]);if(t.a!=null)for(t.b=z(l.W9,C1,105,t.a.length,0,1),e=0;e<t.a.length;e++)t.b[e]=new Tt.YM(t.a[e]);t.d=24;},n.qk=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;for(r=z(l.bab,C1,35,t.k.length,0,1),S=0,M=0,b=0;b<t.k.length;b++)n.be(t.e[b],e,null,q1),r[b]=t.e[b].v,S+=r[b].b,M=j.Math.max(M,r[b].a);for(u=z(l.bab,C1,35,t.a.length,0,1),v=0,C=0,c=0;c<t.a.length;c++)n.be(t.b[c],e,null,65552),u[c]=t.b[c].v,v=j.Math.max(v,u[c].b),C+=u[c].a+12;if(P=j.Math.max(48,v+24),M=j.Math.max(M,24+2*C),A=-1,!t.g)t.g=new n.ek(),U.rj(t.g,new tt.FH()),A=0;else {for(g=0;g<t.g.a.length;g++)if(Jt(U.wj(t.g,g),61)){A=g;break;}A==-1&&(A=t.g.a.length,U.rj(t.g,new tt.FH()));}for(m=18,h=0;h<t.k.length;h++){if(h==t.n){for(tt.EH(U.wj(t.g,A),m-18,M/2,m-18+P,M/2),i=m+-6,D=.5*(M-12)-C,d=0;d<t.a.length;d++)Q=i+.5*(v-u[d].b)-u[d].c,s=D-u[d].d,n.dd(t.b[d],new n.Ci(Q,s)),D+=12+u[d].a;m+=P;}o=m-r[h].c,L=.5*(M-r[h].a)-r[h].d,n.dd(t.e[h],new n.Ci(o,L)),m+=36+r[h].b;}t.o=false;},n.rk=function(t,e){var A,P;if(t.g)for(P=new U.wqb(t.g);P.a<P.c.a.length;)A=U.vqb(P),A.Hb(e,t.p);},n.sk=function(t,e){var A,P,r,i,D;if(t.i!=0&&t.k!=null){for(P=n.nk(t)/t.p.c,R.V2(e,t.i),R.T2(e,Ft(1.6*P),true,false),i=0;i<t.k.length;i++)if(t.k[i].q!=0){for(r=new F1.Rfb(),A=0;A<t.k[i].q;A++)r.a=Ft(r.a+n.Pl(t.k[i],A)),r.b=Ft(r.b+n.Ql(t.k[i],A));r.a=r.a/t.k[i].q|0,r.b=r.b/t.k[i].q|0,r.a=Ft(n.zi(t.e[i].R,r.a)),r.b=Ft(n.Ai(t.e[i].R,r.b)),D=t.c==0?""+(i+1):t.c==2?i<t.j?""+String.fromCharCode(65+i&r1):"R"+(i+1-t.j):t.c==1?i<t.n?""+String.fromCharCode(65+i&r1):"P"+(i+1-t.n):"?"+(i+1),R.C2(e,r.a,r.b,D);}}},n.tk=function(t,e){var A,P,r,i,D;if(t.e!=null)for(A=n.nk(t)/t.p.c,r=t.e,i=0,D=r.length;i<D;++i)P=r[i],n.Wd(P,t.f),P.r=A/P.R.c,n.Td(P,e);if(t.b!=null)for(r=t.b,i=0,D=r.length;i<D;++i)P=r[i],n.Td(P,e);},n.uk=function(t,e){t.d=e;},n.vk=function(t,e){t.f=e;},n.wk=function(t,e){var A,P,r,i;if(t.e!=null)for(P=t.e,r=0,i=P.length;r<i;++r)A=P[r],A.J=e;},n.xk=function(t,e,A){var P,r,i,D,u,o,Q;if(t.e!=null)for(i=t.e,u=0,Q=i.length;u<Q;++u)P=i[u],n.Xd(P,e,A);if(t.b!=null)for(r=t.b,D=0,o=r.length;D<o;++D)P=r[D],n.Xd(P,e,A);},n.yk=function(t,e){t.i=e;},n.zk=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m;if(n.Ak(t,e,A,P),n.yi(t.p))return null;if(t.k!=null)for(h=t.k,b=0,c=h.length;b<c;++b)s=h[b],n.ui(t.p,s);if(t.g)for(d=new U.wqb(t.g);d.a<d.c.a.length;)g=U.vqb(d),n.si(t.p,g);if(t.e!=null)for(D=t.e,o=0,L=D.length;o<L;++o)r=D[o],n.xi(r.R);if(t.b!=null)for(i=t.b,u=0,Q=i.length;u<Q;++u)r=i[u],n.xi(r.R);return m=t.p,t.p=new n.Bi(),m;},n.Ak=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d;if(t.o&&n.qk(t,e),i=null,t.e!=null)for(u=t.e,Q=0,s=u.length;Q<s;++Q)D=u[Q],n.be(D,e,null,0),i=i?Tt.DN(i,D.v):D.v;if(t.b!=null)for(u=t.b,Q=0,s=u.length;Q<s;++Q)D=u[Q],n.be(D,e,null,0),i=i?Tt.DN(i,D.v):D.v;if(t.g)for(c=new U.wqb(t.g);c.a<c.c.a.length;)b=U.vqb(c),g=b.Ib(e),n.wi(t.p,g),i=i?Tt.DN(i,g):g;if(!i)return null;if(r=n.nk(t),d=new n.Di(i,A,r,P),!(d.c==1&&d.a==0&&d.b==0)){if(n.ti(d,t.p),t.e!=null)for(o=t.e,L=0,h=o.length;L<h;++L)D=o[L],n.dd(D,d);if(t.b!=null)for(u=t.b,Q=0,s=u.length;Q<s;++Q)D=u[Q],n.dd(D,d);return d;}return null;},n.Bk=function(t,e){t&&(this.k=z(l.d8,xe,25,1,0,1),this.k[0]=t),this.c=0,this.g=e,n.pk(this);},n.Ck=function(t,e){var A,P;for(this.k=z(l.d8,xe,25,t.g.a.length+t.f.a.length,0,1),P=0;P<t.g.a.length+t.f.a.length;P++)this.k[P]=P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length);for(this.n=t.g.a.length,this.a=z(l.d8,xe,25,t.a.a.length,0,1),A=0;A<t.a.a.length;A++)this.a[A]=U.wj(t.a,A);this.o=false,this.c=1,this.g=e,n.pk(this);},n.Dk=function(t,e,A){this.k=t,this.c=2,this.g=A,this.j=e,n.pk(this);},n.Ek=function(t,e){this.k=t,this.c=0,this.g=e,n.pk(this);},n.Fk=function(t,e){return w.Zzb((w.Lzb(t),t))>w.Zzb((w.Lzb(e),e))?-1:w.Zzb((w.Lzb(t),t))<w.Zzb((w.Lzb(e),e))?1:0;},x(157,1,{},n.Bk,n.Ck,n.Dk,n.Ek),y.c=0,y.d=0,y.f=0,y.i=0,y.j=0,y.n=0,y.o=false,l.D7=X(157),n.Gk=function(){},x(358,1,{},n.Gk),y.Qb=function(t,e){return n.Fk(t,e);},y.lb=function(t){return this===t;},l.C7=X(358),n.Qk=function(){n.Qk=o1,n.Kk=F(k(l.Sbb,1),st,2,6,["?","H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr","Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"R4","R5","R6","R7","R8","R9","R10","R11","R12","R13","R14","R15","R16","R1","R2","R3","A","A1","A2","A3",R1,R1,"D","T","X","R","H2","H+","Nnn","HYD","Pol",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"Ala","Arg","Asn","Asp","Cys","Gln","Glu","Gly","His","Ile","Leu","Lys","Met","Phe","Pro","Ser","Thr","Trp","Tyr","Val"]),n.Ok=F(k(l.yeb,1),Kt,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]),n.Nk=F(k(l._6,1),J,6,15,[6]),n.Jk=F(k(l._6,1),J,6,15,[2]),n.Lk=F(k(l._6,2),Te,10,0,[null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[0]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[0]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[2,4,6]),F(k(l._6,1),J,6,15,[1,3,5,7]),F(k(l._6,1),J,6,15,[0]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,null,null,null,null,null,null,null,null,F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[2,4,6]),F(k(l._6,1),J,6,15,[1,3,5,7]),F(k(l._6,1),J,6,15,[0,2]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,null,null,null,null,null,null,null,null,F(k(l._6,1),J,6,15,[1,2,3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[2,4,6]),F(k(l._6,1),J,6,15,[1,3,5,7]),F(k(l._6,1),J,6,15,[0,2,4,6]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2])]),n.Mk=F(k(l._6,2),Te,10,0,[null,F(k(l._6,1),J,6,15,[1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,F(k(l._6,1),J,6,15,[-3]),F(k(l._6,1),J,6,15,[-2]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),null,F(k(l._6,1),J,6,15,[-3]),F(k(l._6,1),J,6,15,[-2]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3,4]),F(k(l._6,1),J,6,15,[2,3,4,5]),F(k(l._6,1),J,6,15,[2,3,6]),F(k(l._6,1),J,6,15,[2,3,4,7]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[1,2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[-3,3,5]),F(k(l._6,1),J,6,15,[-2]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[6]),F(k(l._6,1),J,6,15,[4,6,7]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[-3,3,5]),F(k(l._6,1),J,6,15,[-2,4,6]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[5]),F(k(l._6,1),J,6,15,[6]),F(k(l._6,1),J,6,15,[4,6,7]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[1,3]),F(k(l._6,1),J,6,15,[1,2]),F(k(l._6,1),J,6,15,[1,3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[-2,2,4]),F(k(l._6,1),J,6,15,[-1,1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[4,5]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[3])]);},n.Rk=function(t,e){var A,P;A=-1,P=0,(t.G[e]&16)!=0&&(A=16,++P),(t.G[e]&64)!=0&&(A=64,++P),(t.G[e]&32)!=0&&(A=32,++P),(t.G[e]&4)!=0&&(A=4,++P),(t.G[e]&2)!=0&&(A=2,++P),(t.G[e]&8)!=0&&(A=8,++P),(t.G[e]&1)!=0&&(A=1,++P),A!=-1&&A!=(t.H[e]&127)&&(t.H[e]=A),P<2&&(t.G[e]&=-128);},n.Sk=function(t,e,A,P){var r;return r=n.Tk(t,6),n.hi(t.J[r],e,A,P),r;},n.Tk=function(t,e){return t.q>=t.M&&n.Qn(t,t.M*2),t.C[t.q]=0,n.Bn(t,t.q,e),t.s[t.q]=0,t.u[t.q]=0,t.B[t.q]=0,t.w[t.q]=0,n.hi(t.J[t.q],0,0,0),t.v!=null&&(t.v[t.q]=null),t.t!=null&&(t.t[t.q]=null),t.T=0,t.q++;},n.Uk=function(t,e,A){return n.Vk(t,e,A,n.Km(t,e)||n.Km(t,A)?16:1);},n.Vk=function(t,e,A,P){var r;if(e==A)return  -1;for(r=0;r<t.r;r++)if(t.D[0][r]==e&&t.D[1][r]==A||t.D[0][r]==A&&t.D[1][r]==e)return t.H[r]<P&&(t.H[r]=P),r;return n.Wk(t,e,A,P);},n.Wk=function(t,e,A,P){return t.r>=t.N&&n.Rn(t,t.N*2),t.D[0][t.r]=e,t.D[1][t.r]=A,t.H[t.r]=P,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},n.Xk=function(t,e){return n.Yk(t,e,e.q,e.r,true);},n.Yk=function(t,e,A,P,r){var i,D,u,o,Q;for(t.K=t.K|e.K,D=z(l.d7,H,6,e.q,15,1),o=n.bn(t,1),Q=n.bn(t,2),i=0;i<A;i++)(r||M1(G(e.B[i],r0),0))&&(D[i]=n.kl(e,t,i,o,Q));for(u=0;u<P;u++)(r||!n.Im(e,e.D[0][u])&&!n.Im(e,e.D[1][u]))&&n.ll(e,t,u,o,Q,D[e.D[0][u]],D[e.D[1][u]],false);return t.L=t.L&&e.L,t.I=0,t.T=0,D;},n.Zk=function(t,e,A,P,r,i,D,u){var o,Q;return o=n.wl(t,e,A),o==-1?(t.q>=t.M&&n.Qn(t,t.M*2),o=n.Tk(t,P),n.hi(t.J[o],e,A,0),t.A[o]=r,n.fn(t,o,i),n.vn(t,o,D),n.ln(t,o,u),true):(Q=n.el(t,o,P,r,i,D),n.ln(t,o,u),Q);},n.$k=function(t,e,A,P){var r;for(r=0;r<t.r;r++)if(t.D[0][r]==e&&t.D[1][r]==A||t.D[0][r]==A&&t.D[1][r]==e)return n.hl(t,r,P),t.T=0,r;return t.r>=t.N&&n.Rn(t,t.N*2),t.D[0][t.r]=e,t.D[1][t.r]=A,t.H[t.r]=P,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},n._k=function(t,e,A,P,r,i){for(var D,u,o;t.q+P>t.M;)n.Qn(t,t.M*2);for(;t.r+P>t.N;)n.Rn(t,t.N*2);return D=n.wl(t,e,A),D!=-1?n.al(t,D,P,r,i):(u=n.xl(t,e,A),u!=-1?n.bl(t,u,P,r,i):(D=n.Sk(t,e,A,0),o=n1*(P-2)/P,n.Vm(t,D,P,D,r,0,n1-o,i),t.T=0,true));},n.al=function(t,e,A,P,r){var i,D,u,o,Q,L;if(P&&n.up(t,e)>1||!P&&n.up(t,e)>2)return  false;for(D=0,i=z(l.b7,jt,6,4,15,1),o=0;o<t.r;o++){for(Q=0;Q<2;Q++)if(t.D[Q][o]==e){if(D==2){D=3;break;}i[D++]=n.Xl(t,e,t.D[1-Q][o]);}if(D==3)break;}return D==3?false:(L=D==1?i[0]+n1:j.Math.abs(i[0]-i[1])>n1?(i[0]+i[1])/2:(i[0]+i[1])/2+n1,u=n1*(A-2)/A,n.Vm(t,e,A,e,P,L-u/2,n1-u,r),t.T=0,true);},n.bl=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c,g;if(Q=z(l.d7,H,6,2,15,1),o=z(l.b7,jt,6,2,15,1),Q[0]=t.D[0][e],Q[1]=t.D[1][e],n.up(t,Q[0])>3||n.up(t,Q[1])>3)return  false;for(D=0,i=z(l.b7,jt,6,4,15,1),h=0;h<t.r;h++)if(h!=e){for(b=0;b<2;b++){for(c=0;c<2;c++)if(t.D[b][h]==Q[c]){if(D==4){D=5;break;}i[D++]=n.Xl(t,Q[c],t.D[1-b][h]);}if(D==5)break;}if(D==5)break;}if(D==5)return  false;for(o[0]=n.Xl(t,Q[0],Q[1]),o[0]<0?(o[1]=o[0]+n1,u=0):(o[1]=o[0],o[0]=o[1]-n1,u=1),g=0,s=0;s<D;s++)i[s]>o[0]&&i[s]<o[1]?--g:++g;return u=g>0?1-u:u,L=n1*(A-2)/A,n.Vm(t,Q[u],A-1,Q[1-u],P,o[g>0?0:1]+n1-L,n1-L,r),t.T=0,true;},n.cl=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g;for(r=z(l.d7,H,6,e.q,15,1),o=n.bn(t,1),Q=n.bn(t,2),P=0;P<e.q;P++)if(e.C[P]!=0||(e.t==null||e.t[P]==null?null:T.cjb((b=e.t[P],V1.Hmb(),b)))!=null)r[P]=n.kl(e,t,P,o,Q);else if(r[P]=A,(e.T&8)!=0){for(u=0;u<e.r;u++)for(L=0;L<2;L++)if(e.D[L][u]==P){if(c=e.D[1-L][u],g=e.u[c]&3,s=false,g==1||g==2){for(i=0;i<e.r;i++)if(i!=u)for(h=0;h<2;h++)e.D[h][i]==c&&e.D[1-h][i]<P&&(s=!s);}s&&n.tn(e,c,g==1?2:1,(e.u[c]&4)!=0);}}for(D=0;D<e.r;D++)n.ll(e,t,D,o,Q,r[e.D[0][D]],r[e.D[1][D]],false);return t.L=t.L&&e.L,t.I=0,t.T=0,r;},n.dl=function(t,e){var A,P,r,i,D,u,o,Q,L;return A=t.J[e[0]],P=t.J[e[1]],r=t.J[e[2]],i=t.J[e[3]],o=new n.ni(P.a-A.a,P.b-A.b,P.c-A.c),Q=new n.ni(r.a-P.a,r.b-P.b,r.c-P.c),L=new n.ni(i.a-r.a,i.b-r.b,i.c-r.c),D=new n.ni(o.b*Q.c-o.c*Q.b,-(o.a*Q.c-o.c*Q.a),o.a*Q.b-o.b*Q.a),u=new n.ni(Q.b*L.c-Q.c*L.b,-(Q.a*L.c-Q.c*L.a),Q.a*L.b-Q.b*L.a),-j.Math.atan2(j.Math.sqrt(Q.a*Q.a+Q.b*Q.b+Q.c*Q.c)*(o.a*u.a+o.b*u.b+o.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},n.el=function(t,e,A,P,r,i){return (A==1||A==151||A==152)&&n.up(t,e)>1||(t.B[e]=G(t.B[e],-2),t.v!=null&&(t.v[e]=null),t.t!=null&&(t.t[e]=null),A==t.C[e]&&P==t.A[e]&&r==((t.u[e]&be)>>>27)-1&&i==(t.u[e]&48))?false:((A==151||A==152)&&(P=A-149,A=1),t.u[e]&=960,t.C[e]=A,t.A[e]=P,t.s[e]=0,t.B[e]=0,n.fn(t,e,r),n.vn(t,e,i),n._m(t,t.w[e]),t.T=0,true);},n.fl=function(t,e,A,P){var r;return r=n.wl(t,e,A),r!=-1&&n.gl(t,r,P);},n.gl=function(t,e,A){if(A){if(t.s[e]>8)return  false;++t.s[e];}else {if(t.s[e]<-8)return  false;--t.s[e];}return t.T=0,true;},n.hl=function(t,e,A){var P,r,i,D,u;return i=false,D=t.H[e],A==511?i=n.rm(t,e):n.Dq(t,e,A)&&(A==257||A==129?(P=n.Wm(t,e,t.D[0][e]),r=n.Wm(t,e,t.D[1][e]),A==D?(P==r||r)&&(u=t.D[0][e],t.D[0][e]=t.D[1][e],t.D[1][e]=u,i=true):(!P&&r&&(u=t.D[0][e],t.D[0][e]=t.D[1][e],t.D[1][e]=u),t.H[e]=A,i=true)):(t.H[e]=A,i=true)),i&&(t.T=(D&127)==(A&127)?t.T&7:0,t.G[e]=0),i;},n.il=function(t){t.q=0,t.r=0,t.K=false,t.L=false,t.I=0,t.v=null,t.t=null,t.P=null,t.T=0;},n.jl=function(t){var e,A,P,r,i,D,u,o;for(D=0;D<t.r;D++)t.H[D]==512&&(A=t.D[0][D],P=t.D[1][D],t.C[A]==-1^t.C[P]==-1&&t.s[A]!=0&&t.s[P]!=0&&t.s[A]<0^t.s[P]<0&&(t.s[A]<0?(++t.s[A],--t.s[P]):(--t.s[A],++t.s[P])));for(o=z(l.d7,H,6,t.q,15,1),r=0,e=0;e<t.q;e++){if(t.C[e]==-1){o[e]=-1;continue;}r<e&&(t.C[r]=t.C[e],t.s[r]=t.s[e],t.A[r]=t.A[e],t.u[r]=t.u[e],t.B[r]=t.B[e],t.w[r]=t.w[e],n.ii(t.J[r],t.J[e]),t.v!=null&&(t.v[r]=t.v[e]),t.t!=null&&(t.t[r]=t.t[e])),o[e]=r,++r;}for(t.q=r,u=0,i=0;i<t.r;i++)t.H[i]!=512&&(t.H[u]=t.H[i],t.F[u]=t.F[i],t.G[u]=t.G[i],t.D[0][u]=o[t.D[0][i]],t.D[1][u]=o[t.D[1][i]],++u);return t.r=u,o;},n.kl=function(t,e,A,P,r){var i,D,u;return i=e.q,i>=e.M&&n.Qn(e,e.M*2),u=(t.u[A]&U1)>>18,D=-1,u==1?P==-1?D=n.bn(e,u):D=j.Math.min(31,P+((t.u[A]&U1)>>18!=1&&(t.u[A]&U1)>>18!=2?-1:(t.u[A]&k0)>>20)):u==2&&(r==-1?D=n.bn(e,u):D=j.Math.min(31,r+((t.u[A]&U1)>>18!=1&&(t.u[A]&U1)>>18!=2?-1:(t.u[A]&k0)>>20))),e.C[i]=t.C[A],e.s[i]=t.s[A],e.A[i]=t.A[A],e.u[i]=t.u[A],e.B[i]=e.K?t.B[A]:0,n.ii(e.J[i],t.J[A]),e.w[i]=t.w[A],e.v!=null&&(e.v[i]=null),t.v!=null&&t.v[A]!=null&&e.K&&(e.v==null&&(e.v=z(l.d7,S1,7,e.C.length,0,2)),e.v[i]=U.Eqb(t.v[A],t.v[A].length)),e.t!=null&&(e.t[i]=null),t.t!=null&&t.t[A]!=null&&(e.t==null&&(e.t=z(l._6,Te,10,e.C.length,0,2)),e.t[i]=U.Cqb(t.t[A],t.t[A].length)),D!=-1&&(e.u[i]&=-32505857,e.u[i]|=D<<20),++e.q,e.T=0,i;},n.ll=function(t,e,A,P,r,i,D,u){var o,Q,L,s;return Q=e.r,Q>=e.N&&n.Rn(e,e.N*2),s=(t.F[A]&768)>>8,L=-1,s==1&&(P==-1?L=n.bn(e,s):L=j.Math.min(32,P+((t.F[A]&768)>>8!=1&&(t.F[A]&768)>>8!=2?-1:(t.F[A]&S0)>>10))),s==2&&(r==-1?L=n.bn(e,s):L=j.Math.min(32,r+((t.F[A]&768)>>8!=1&&(t.F[A]&768)>>8!=2?-1:(t.F[A]&S0)>>10))),e.D[0][Q]=i,e.D[1][Q]=D,o=u&&t.Yb(A)?8:t.H[A],e.H[Q]=o,e.F[Q]=t.F[A],e.G[Q]=e.K?t.G[A]:0,L!=-1&&(e.F[Q]&=-31745,e.F[Q]|=L<<10),++e.r,e.T=0,Q;},n.ml=function(t,e,A,P,r,i,D){return n.ll(t,e,A,P,r,i==null?t.D[0][A]:i[t.D[0][A]],i==null?t.D[1][A]:i[t.D[1][A]],D);},n.nl=function(t,e){var A,P;for(e.v=null,e.t=null,e.K=t.K,e.q=0,A=0;A<t.q;A++)n.kl(t,e,A,0,0);for(e.r=0,P=0;P<t.r;P++)n.ll(t,e,P,0,0,t.D[0][P],t.D[1][P],false);t.Xb(e);},n.ol=function(t,e){e.K=t.K,e.L=t.L,e.S=t.S,e.I=t.I,e.P=t.P,e.T=t.T&24;},n.pl=function(t,e){var A,P;for(A=0;A<t.r;A++)for(P=0;P<2;P++)t.D[P][A]==e&&(t.H[A]=512);n._m(t,t.w[e]),t.C[e]=-1,t.v!=null&&(t.v[e]=null),t.t!=null&&(t.t[e]=null),n.jl(t),t.T=0;},n.ql=function(t,e,A){var P,r;return P=n.wl(t,e,A),P!=-1?((t.u[P]&512)!=0?n.vl(t):n.pl(t,P),t.T=0,true):(r=n.xl(t,e,A),r!=-1?((t.u[t.D[0][r]]&t.u[t.D[1][r]]&512)!=0?n.vl(t):n.tl(t,r),t.T=0,true):false);},n.rl=function(t,e){var A,P,r,i;if(e.length==0)return null;for(P=e,r=0,i=P.length;r<i;++r)A=P[r],t.C[A]=-1;return n.ul(t);},n.sl=function(t,e){t.H[e]=512,n.jl(t),t.T=0;},n.tl=function(t,e){var A,P,r;for(P=0;P<2;P++){for(A=0,r=0;r<t.r;r++)r!=e&&(t.D[0][r]==t.D[P][e]||t.D[1][r]==t.D[P][e])&&++A;A==0&&(n._m(t,t.w[t.D[P][e]]),t.C[t.D[P][e]]=-1);}t.H[e]=512,n.jl(t),t.T=0;},n.ul=function(t){var e,A,P;for(P=false,e=0;e<t.q;e++)t.C[e]==-1&&(P=true,n._m(t,t.w[e]));for(A=0;A<t.r;A++)t.H[A]==512?P=true:(t.C[t.D[0][A]]==-1||t.C[t.D[1][A]]==-1)&&(t.H[A]=512,P=true);return P?(t.T=0,n.jl(t)):null;},n.vl=function(t){var e,A;for(A=false,e=0;e<t.q;e++)(t.u[e]&512)!=0&&(t.C[e]=-1,A=true);return A&&n.ul(t)!=null;},n.wl=function(t,e,A){var P,r,i,D,u,o,Q,L;for(D=-1,r=n.Vl(t,t.q,t.r,n.Pk),u=L0,o=r*r/12,P=0;P<t.q;P++)Q=t.J[P].a,L=t.J[P].b,i=(e-Q)*(e-Q)+(A-L)*(A-L),i<o&&i<u&&(u=i,D=P);return D;},n.xl=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;for(h=-1,c=n.Vl(t,t.q,t.r,n.Pk),b=L0,P=0;P<t.r;P++)g=t.J[t.D[0][P]].a,m=t.J[t.D[0][P]].b,d=t.J[t.D[1][P]].a,C=t.J[t.D[1][P]].b,L=d-g,s=C-m,r=j.Math.sqrt(L*L+s*s),i=(g+d)/2,D=(m+C)/2,L=e-i,s=A-D,!(j.Math.sqrt(L*L+s*s)>r/2)&&(d==g?Q=j.Math.abs(g-e):(u=(C-m)/(g-d),o=-u*g-m,Q=j.Math.abs((u*e+A+o)/j.Math.sqrt(u*u+1))),Q<c&&Q<b&&(b=Q,h=P));return h;},n.yl=function(t,e){return ((t.u[e]&be)>>>27)-1;},n.zl=function(t,e){return (t.u[e]&49152)>>14;},n.Al=function(t,e){return t.s[e];},n.Bl=function(t,e){return t.u[e]&448;},n.Cl=function(t,e){return t.J[e];},n.Dl=function(t,e){var A;return t.t==null||t.t[e]==null?null:T.cjb((A=t.t[e],V1.Hmb(),A));},n.El=function(t,e){return t.t==null?null:t.t[e];},n.Fl=function(t,e){return (t.u[e]&U1)>>18!=1&&(t.u[e]&U1)>>18!=2?-1:(t.u[e]&k0)>>20;},n.Gl=function(t,e){return (t.u[e]&U1)>>18;},n.Hl=function(t,e){return n.Kk[t.C[e]];},n.Il=function(t,e){return t.v==null?null:t.v[e];},n.Jl=function(t,e){var A,P,r;if(t.v==null||t.v[e]==null)return kt(G(t.B[e],1),0)?"":n.Kk[t.C[e]];for(r="",P=0;P<t.v[e].length;P++)P>0&&(r=(w.Lzb(r),r+(w.Lzb(","),","))),A=t.v[e][P],r=T.ajb(r,n.Kk[A]);return r;},n.Kl=function(t,e){return j.Math.abs(t.w[e]);},n.Ll=function(t,e){return t.A[e];},n.Ml=function(t,e){return t.u[e]&3;},n.Nl=function(t,e){return t.B[e];},n.Ol=function(t,e){return t.u[e]&48;},n.Pl=function(t,e){return t.J[e].a;},n.Ql=function(t,e){return t.J[e].b;},n.Rl=function(t,e){return t.J[e].c;},n.Sl=function(t,e){return t.C[e];},n.Tl=function(t){return n.Vl(t,t.q,t.r,n.Pk);},n.Ul=function(t,e){return n.Vl(t,t.q,t.r,e);},n.Vl=function(t,e,A,P){return n.Wl(t,e,A,P,t.J);},n.Wl=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c;for(s=false,h=0,Q=0;Q<A;Q++)t.H[Q]!=16&&(t.G[Q]&Le)==0&&++h;if(h==0){for(L=0;L<A;L++)(t.G[L]&Le)==0&&++h;s=true;}if(h==0){if(e<2)return P;for(c=L0,i=1;i<e;i++)for(D=0;D<i;D++)b=n.ci(r[i],r[D]),b>0&&b<c&&(c=b);return c!=L0?.6*c:P;}for(u=0,o=0;o<A;o++)(s||t.H[o]!=16)&&(t.G[o]&Le)==0&&(u+=n.ci(r[t.D[1][o]],r[t.D[0][o]]));return u/h;},n.Xl=function(t,e,A){return n.mo(t.J[e].a,t.J[e].b,t.J[A].a,t.J[A].b);},n.Yl=function(t,e,A){return t.D[e][A];},n.Zl=function(t,e){return ((t.G[e]&7680)>>9)+((t.G[e]&$A)>>13);},n.$l=function(t,e){return (t.G[e]&7680)>>9;},n._l=function(t,e){return (t.F[e]&48)>>4;},n.am=function(t,e){return (t.F[e]&768)>>8!=1&&(t.F[e]&768)>>8!=2?-1:(t.F[e]&S0)>>10;},n.bm=function(t,e){return (t.F[e]&768)>>8;},n.cm=function(t,e){var A,P,r,i;return A=t.D[0][e],P=t.D[1][e],r=t.J[P].a-t.J[A].a,i=t.J[P].b-t.J[A].b,j.Math.sqrt(r*r+i*i);},n.dm=function(t,e,A){var P;for(P=0;P<t.r;P++)if((t.D[0][P]==e&&t.D[1][P]==A||t.D[0][P]==A&&t.D[1][P]==e)&&t.H[P]!=512)return P;return  -1;},n.em=function(t,e){if(t.K&&(t.G[e]&127)!=0){if((t.G[e]&9)!=0)return 1;if((t.G[e]&2)!=0)return 2;if((t.G[e]&4)!=0)return 3;if((t.G[e]&32)!=0)return 4;if((t.G[e]&64)!=0)return 5;if((t.G[e]&16)!=0)return 0;}switch(t.H[e]&127){case 1:case 8:return 1;case 2:return 2;case 4:return 3;case 32:return 4;case 64:return 5;default:return 0;}},n.fm=function(t,e){return t.F[e]&3;},n.gm=function(t,e){return t.G[e];},n.hm=function(t,e){return t.H[e];},n.im=function(t,e){return t.H[e]&127;},n.jm=function(t,e){var A,P,r,i,D;if(t.q==0)return null;for(P=t.J[0].a,i=t.J[0].b,r=t.J[0].a,D=t.J[0].b,A=1;A<t.q;A++)P>t.J[A].a?P=t.J[A].a:r<t.J[A].a&&(r=t.J[A].a),i>t.J[A].b?i=t.J[A].b:D<t.J[A].b&&(D=t.J[A].b);return e?(e.c=P,e.d=i,e.b=r-P,e.a=D-i):e=new Tt.FN(P,i,r-P,D-i),e;},n.km=function(t,e){var A;return A=t.C[e]<n.Lk.length?n.Lk[t.C[e]]:null,A==null?6:A[A.length-1];},n.lm=function(t,e,A){return n.mm(t,e,A,t.C[e]);},n.mm=function(t,e,A,P){var r,i;return P>=171&&P<=190?0:(i=0,(t.u[e]&48)==32&&(i-=1),((t.u[e]&48)==16||(t.u[e]&48)==48)&&(i-=2),r=t.s[e],r==0&&t.K&&(M1(G(t.B[e],je),ln)&&(r=-1),M1(G(t.B[e],je),on)&&(r=1)),P==7||P==8||P==9?i+=r:P==6||P==14||P==32?i-=j.Math.abs(r):P==15||P==33?A-i-r<=3?i+=r:i-=r:P==16||P==34||P==52?A-i-r<=4?i+=r:i-=j.Math.abs(r):P==17||P==35||P==53?A-i-r<=5?i+=r:i-=j.Math.abs(r):i-=r,i);},n.nm=function(t,e){var A;return A=n.om(t,e),A+n.mm(t,e,A,t.C[e]);},n.om=function(t,e){var A,P;return A=((t.u[e]&be)>>>27)-1,A==-1&&(A=(P=t.C[e]<n.Lk.length?n.Lk[t.C[e]]:null,P==null?6:P[P.length-1])),A;},n.pm=function(t,e){var A,P,r,i,D;for(i=n.Sm(t,t.D[0][e])||n.Sm(t,t.D[1][e])?5:3,P=0;P<2;P++)A=t.D[P][e],r=n.em(t,e)+(D=n.om(t,A),D+n.mm(t,A,D,t.C[A]))-n.up(t,A),i>r&&(i=r);return i;},n.qm=function(t,e){return (t.u[e]&q1)!=0;},n.rm=function(t,e){var A,P,r;return P=n.pm(t,e),A=n.Km(t,t.D[0][e])||n.Km(t,t.D[1][e]),r=A?16:1,t.H[e]==64?(t.H[e]=r,t.T=0,true):t.H[e]==32?(t.H[e]=P>4?64:r,t.T=0,true):t.H[e]==4?(t.H[e]=P>3?32:r,t.T=0,true):t.H[e]==2&&(t.H[e]=386,t.T&=7,(t.F[e]&128)==0)?true:t.H[e]==386?(P>2?t.H[e]=4:t.H[e]=r,t.T=0,true):(384&t.H[e])!=0?(t.H[e]=1,t.T&=7,true):!A&&P<2?false:t.H[e]==1?(t.H[e]=2,t.T=0,true):P<1?false:t.H[e]==16?(t.H[e]=1,t.T=0,true):false;},n.sm=function(t){var e;for(t.T=0,t.C=z(l.d7,H,6,t.M,15,1),t.s=z(l.d7,H,6,t.M,15,1),t.w=z(l.d7,H,6,t.M,15,1),t.J=z(l.w7,Sn,29,t.M,0,1),e=0;e<t.M;e++)t.J[e]=new n.mi();t.A=z(l.d7,H,6,t.M,15,1),t.u=z(l.d7,H,6,t.M,15,1),t.B=z(l.e7,n0,6,t.M,14,1),t.v=null,t.t=null,t.D=h1(l.d7,[S1,H],[7,6],15,[2,t.N],2),t.H=z(l.d7,H,6,t.N,15,1),t.F=z(l.d7,H,6,t.N,15,1),t.G=z(l.d7,H,6,t.N,15,1);},n.tm=function(t,e){t.T&=~e;},n.um=function(t){var e;for(e=0;e<t.q;e++)if(t.J[e].c!=0)return  true;return  false;},n.vm=function(t,e){return (t.u[e]&lA)!=0;},n.wm=function(t,e){return t.C[e]==-1;},n.xm=function(t,e){return (t.u[e]&4)!=0;},n.ym=function(t,e){return (t.u[e]&q0)!=0;},n.zm=function(t,e){return t.w[e]<0;},n.Am=function(t,e){return (t.F[e]&Oe)!=0;},n.Bm=function(t,e){return (t.G[e]&Le)!=0;},n.Cm=function(t,e){return (t.F[e]&q1)!=0;},n.Dm=function(t,e){return t.H[e]==512;},n.Em=function(t,e){return (t.F[e]&4)!=0;},n.Fm=function(t,e){return (t.F[e]&A0)!=0;},n.Gm=function(t,e){var A,P,r,i;if(t.K){if(kt(G(t.B[e],1),0))return  false;if(t.v!=null&&t.v[e]!=null){for(P=t.v[e],r=0,i=P.length;r<i;++r)if(A=P[r],!n.po(A))return  false;}}return n.po(t.C[e]);},n.Hm=function(t,e){var A,P,r,i;if(t.K){if(kt(G(t.B[e],1),0))return  false;if(t.v!=null&&t.v[e]!=null){for(P=t.v[e],r=0,i=P.length;r<i;++r)if(A=P[r],!n.qo(A))return  false;}}return n.qo(t.C[e]);},n.Im=function(t,e){return kt(G(t.B[e],r0),0);},n.Jm=function(t,e){return (t.u[e]&A0)!=0;},n.Km=function(t,e){var A,P,r,i;if(t.K){if(kt(G(t.B[e],1),0))return  false;if(t.v!=null&&t.v[e]!=null){for(P=t.v[e],r=0,i=P.length;r<i;++r)if(A=P[r],!(A>=3&&A<=4||A>=11&&A<=13||A>=19&&A<=31||A>=37&&A<=51||A>=55&&A<=84||A>=87&&A<=112))return  false;}}return n.ro(t.C[e]);},n.Lm=function(t,e){return t.A[e]==0;},n.Mm=function(t,e){var A,P,r,i;if(t.K){if(kt(G(t.B[e],1),0))return  false;if(t.v!=null&&t.v[e]!=null){for(P=t.v[e],r=0,i=P.length;r<i;++r)if(A=P[r],!(A==1||A>=5&&A<=9||A>=14&&A<=17||A>=32&&A<=35||A>=52&&A<=53))return  false;}}return n.to(t.C[e]);},n.Nm=function(t){var e;for(e=0;e<t.q;e++)switch(t.C[e]){case 1:case 5:case 6:case 7:case 8:case 9:case 14:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:continue;default:return  false;}return  true;},n.Om=function(t,e){return (t.u[e]&512)!=0;},n.Pm=function(t,e){return (t.u[t.D[0][e]]&t.u[t.D[1][e]]&512)!=0;},n.Qm=function(t,e){return t.H[e]==257||t.H[e]==129;},n.Rm=function(t,e,A){return (t.H[e]==257||t.H[e]==129)&&t.D[0][e]==A;},n.Sm=function(t,e){var A,P,r,i;if(t.K){if(kt(G(t.B[e],1),0))return  false;if(t.v!=null&&t.v[e]!=null){for(P=t.v[e],r=0,i=P.length;r<i;++r)if(A=P[r],!(A>=21&&A<=30||A>=39&&A<=48||A==57||A>=72&&A<=80||A==89||A>=104&&A<=112))return  false;}}return n.uo(t.C[e]);},n.Tm=function(t,e){t.C[e]=-1;},n.Um=function(t,e){t.H[e]=512;},n.Vm=function(t,e,A,P,r,i,D,u){var o,Q,L,s,h,b,c,g,d,m;for(e!=P&&(d=t.J[e].a-t.J[P].a,m=t.J[e].b-t.J[P].b,u=j.Math.sqrt(d*d+m*m)),o=e,L=n.Vn(t,e)!=3,g=1;g<A;g++){for(h=t.J[o].a+u*j.Math.sin(i),b=t.J[o].b+u*j.Math.cos(i),c=-1,s=0;s<t.q;s++)if(j.Math.abs(h-t.J[s].a)<4&&j.Math.abs(b-t.J[s].b)<4){c=s;break;}c==-1&&(c=n.Sk(t,h,b,0),t.J[c].a=h,t.J[c].b=b,t.J[c].c=0),Q=n.dm(t,o,c),Q==-1&&(Q=n.Vk(t,o,c,n.Km(t,o)||n.Km(t,c)?16:1),r&&(L&&n.Vn(t,t.D[0][Q])<4&&n.Vn(t,t.D[1][Q])<3&&(t.H[Q]=2),L=!L)),o=c,i+=D;}Q=n.dm(t,o,P),Q==-1&&(Q=n.Vk(t,o,P,n.Km(t,o)||n.Km(t,P)?16:1)),r&&L&&n.Vn(t,t.D[0][Q])<4&&n.Vn(t,t.D[1][Q])<4&&(t.H[Q]=2);},n.Wm=function(t,e,A){var P,r;if(n.em(t,e)!=1)return  false;if((t.u[A]&3)!=0)return  true;for(r=0;r<t.r;r++)if(r!=e&&t.H[r]==2&&(t.D[0][r]==A&&(t.u[t.D[1][r]]&3)!=0||t.D[1][r]==A&&(t.u[t.D[0][r]]&3)!=0))return  true;for(P=0;P<t.r;P++)if(P!=e&&t.H[P]==1&&(t.D[0][P]==A||t.D[1][P]==A)&&(t.F[P]&3)!=0)return  true;return  false;},n.Xm=function(t){var e;for(e=0;e<t.q;e++)t.u[e]&=-449;},n.Ym=function(t){var e;for(e=0;e<t.q;e++)t.u[e]&=aA;},n.Zm=function(t){var e;for(e=0;e<t.q;e++)t.u[e]&=-513;},n.$m=function(t){var e;for(e=0;e<t.r;e++)t.F[e]&=-98305;},n._m=function(t,e){var A;for(A=0;A<t.q;A++)j.Math.abs(t.w[A])==j.Math.abs(e)&&(t.w[A]=0);},n.an=function(t){var e,A,P,r;for(r=false,A=0;A<t.q;A++)kt(G(t.B[A],r0),0)&&(t.C[A]=-1,r=true);for(r&&n.ul(t),t.v!=null&&(t.v=null,r=true),e=0;e<t.q;e++)kt(t.B[e],0)&&(t.B[e]=0,r=true);for(P=0;P<t.r;P++)t.G[P]!=0&&(t.G[P]=0,r=true),t.H[P]==8&&(t.H[P]=1,r=true);return r&&(t.T=0),r;},n.bn=function(t,e){var A,P,r,i,D,u,o,Q,L;if(e==0)return 0;for(u=null,P=0;P<t.q;P++)(t.u[P]&U1)>>18==e&&(u==null&&(u=z(l.zeb,ht,6,32,16,1)),u[(t.u[P]&U1)>>18!=1&&(t.u[P]&U1)>>18!=2?-1:(t.u[P]&k0)>>20]=true);for(i=0;i<t.r;i++)(t.F[i]&768)>>8==e&&(u==null&&(u=z(l.zeb,ht,6,32,16,1)),u[(t.F[i]&768)>>8!=1&&(t.F[i]&768)>>8!=2?-1:(t.F[i]&S0)>>10]=true);if(L=0,u!=null){for(Q=z(l.d7,H,6,32,15,1),o=0;o<32;o++)u[o]&&(Q[o]=L++);for(A=0;A<t.q;A++)(t.u[A]&U1)>>18==e&&(D=Q[(t.u[A]&U1)>>18!=1&&(t.u[A]&U1)>>18!=2?-1:(t.u[A]&k0)>>20],t.u[A]&=-32505857,t.u[A]|=D<<20);for(r=0;r<t.r;r++)(t.F[r]&768)>>8==e&&(D=Q[(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&S0)>>10],t.F[r]&=-31745,t.F[r]|=D<<10);}return L;},n.cn=function(t,e){var A;for(A=0;A<t.q;A++)t.J[A].a*=e,t.J[A].b*=e;},n.dn=function(t,e){t.q=e,t.T=0;},n.en=function(t,e){t.r=e,t.T=0;},n.fn=function(t,e,A){A>=-1&&A<=14&&(t.u[e]&=-2013265921,t.u[e]|=1+A<<27,t.C[e]==6&&(A==-1||A==0||A==2||A==4)&&(t.u[e]&=-49,A==2&&(t.u[e]|=16)));},n.gn=function(t,e,A){t.u[e]&=-49153,t.u[e]|=A<<14;},n.hn=function(t,e,A){t.s[e]=A,t.T=0;},n.jn=function(t,e,A){t.u[e]&=-449,t.u[e]|=A;},n.kn=function(t,e,A){A?t.u[e]|=lA:t.u[e]&=tP,t.T&=7;},n.ln=function(t,e,A){var P,r;A!=null&&(w.Xzb(A).length==0?A=null:(P=n.oo(A,321),(P!=0&&P==t.C[e]||T.ejb(A,"?"))&&(n.Bn(t,e,P),A=null))),A==null?t.t!=null&&(t.t[e]=null):(t.t==null&&(t.t=z(l._6,Te,10,t.M,0,2)),t.t[e]=w.xzb((r=A,V1.Hmb(),r)));},n.mn=function(t,e,A){A!=null&&A.length==0&&(A=null),A==null?t.t!=null&&(t.t[e]=null):(t.t==null&&(t.t=z(l._6,Te,10,t.M,0,2)),t.t[e]=A);},n.nn=function(t,e,A,P){var r,i,D;if(A==0)t.u[e]&=eP,t.u[e]|=A<<18;else {if(P>=32)return;if(P==-1){for(D=-1,i=0;i<t.q;i++)i!=e&&A==(t.u[i]&U1)>>18&&D<((t.u[i]&U1)>>18!=1&&(t.u[i]&U1)>>18!=2?-1:(t.u[i]&k0)>>20)&&(D=(t.u[i]&U1)>>18!=1&&(t.u[i]&U1)>>18!=2?-1:(t.u[i]&k0)>>20);for(r=0;r<t.r;r++)A==(t.F[r]&768)>>8&&D<((t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&S0)>>10)&&(D=(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&S0)>>10);if(P=D+1,P>=32)return;}t.u[e]&=eP,t.u[e]|=A<<18|P<<20;}t.T&=7;},n.on=function(t,e,A){t.v==null&&(t.v=z(l.d7,S1,7,t.M,0,2)),w.qzb(A,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),t.v[e]=A,t.T=0,t.K=true;},n.pn=function(t,e,A,P){var r;if(A==null){t.v!=null&&(t.v[e]=null);return;}if(A.length==1&&!P){r=A[0],t.C[e]!=r&&n.el(t,e,r,0,-1,0),t.v!=null&&(t.v[e]=null);return;}t.v==null&&(t.v=z(l.d7,S1,7,t.M,0,2)),t.v[e]=A,P&&(t.B[e]=Pt(t.B[e],1)),t.T=0,t.K=true;},n.qn=function(t,e,A,P){t.w[e]=P?-A:A;},n.rn=function(t,e,A){A?t.u[e]|=A0:t.u[e]&=aA;},n.sn=function(t,e,A){t.A[e]=A,t.T&=t.C[e]==1?0:7;},n.tn=function(t,e,A,P){t.u[e]&=-8,A!=3&&(t.u[e]&=tP),t.u[e]|=A,P&&(t.u[e]|=4);},n.un=function(t,e,A,P){P?t.B[e]=Pt(t.B[e],A):t.B[e]=G(t.B[e],te(A)),t.T=0,t.K=true;},n.vn=function(t,e,A){t.u[e]&=-49,t.u[e]|=A,t.T&=7;},n.wn=function(t,e,A){A?t.u[e]|=512:t.u[e]&=-513;},n.xn=function(t,e,A){t.u[e]&=-67108865,A&&(t.u[e]|=q0);},n.yn=function(t,e,A){t.J[e].a=A,t.T&=7;},n.zn=function(t,e,A){t.J[e].b=A,t.T&=7;},n.An=function(t,e,A){t.J[e].c=A,t.T&=7;},n.Bn=function(t,e,A){A>=0&&A<=190&&(A==151||A==152?(t.C[e]=1,t.A[e]=A-149):(t.C[e]=A,t.A[e]=0),t.u[e]&=-2013265921,t.T=0);},n.Cn=function(t,e,A,P){t.D[e][A]=P,t.T=0;},n.Dn=function(t,e,A){A?t.F[e]|=Oe:t.F[e]&=-32769;},n.En=function(t,e,A){t.F[e]&=-49,t.F[e]|=A<<4;},n.Fn=function(t,e,A,P){var r,i,D;if(A==0)t.F[e]&=-32513,t.F[e]|=A<<8;else {if(P>=32)return;if(P==-1){for(D=-1,i=0;i<t.q;i++)A==(t.u[i]&U1)>>18&&D<((t.u[i]&U1)>>18!=1&&(t.u[i]&U1)>>18!=2?-1:(t.u[i]&k0)>>20)&&(D=(t.u[i]&U1)>>18!=1&&(t.u[i]&U1)>>18!=2?-1:(t.u[i]&k0)>>20);for(r=0;r<t.r;r++)r!=e&&A==(t.F[r]&768)>>8&&D<((t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&S0)>>10)&&(D=(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&S0)>>10);if(P=D+1,P>=32)return;}t.F[e]&=-32513,t.F[e]|=A<<8|P<<10;}t.T&=7;},n.Gn=function(t,e,A){A?t.F[e]|=q1:t.F[e]&=-65537;},n.Hn=function(t,e,A){t.H[e]=A==1?1:A==2?2:A==3?4:16,t.T=0;},n.In=function(t,e,A,P){t.F[e]&=-131080,t.F[e]|=A,P&&(t.F[e]|=4);},n.Jn=function(t,e){t.F[e]|=A0;},n.Kn=function(t,e,A,P){P?t.G[e]|=A:t.G[e]&=~A,t.T=0,P&&A!=0&&(t.K=true);},n.Ln=function(t,e,A){t.H[e]=A,t.T=0;},n.Mn=function(t,e){t.I=e;},n.Nn=function(t,e){t.K!=e&&(t.K=e,e||n.an(t),t.T=0);},n.On=function(t,e){t.T=e;},n.Pn=function(t,e){t.S=e;},n.Qn=function(t,e){var A,P;for(t.C=n.go(t.C,e),t.s=n.go(t.s,e),t.w=n.go(t.w,e),P=t.J.length,t.J=n.io(t.J,e),A=P;A<e;A++)t.J[A]=new n.mi();t.A=n.go(t.A,e),t.u=n.go(t.u,e),t.B=n.ho(t.B,e),t.v!=null&&(t.v=n.ko(t.v,e)),t.t!=null&&(t.t=n.jo(t.t,e)),t.M=e;},n.Rn=function(t,e){t.D[0]=n.go(t.D[0],e),t.D[1]=n.go(t.D[1],e),t.H=n.go(t.H,e),t.F=n.go(t.F,e),t.G=n.go(t.G,e),t.N=e;},n.Sn=function(t,e){t.O=e;},n.Tn=function(t,e){t.P=e;},n.Un=function(t,e){t.u[e]|=q1;},n.Vn=function(t,e){var A,P;for(P=0,A=0;A<t.r;A++)(t.D[0][A]==e||t.D[1][A]==e)&&(P+=n.em(t,A));return P;},n.Wn=function(t){var e,A,P;for(A=false,P=false,e=0;e<t.q;e++)t.A[e]!=0&&(t.A[e]=0,A=true,t.C[e]==1&&(P=true));return P&&(t.T=0),A;},n.Xn=function(t,e,A){return n.Km(t,e)||n.Km(t,A)?16:1;},n.Yn=function(t,e,A){var P,r,i,D,u,o;for(D=t.C[e],t.C[e]=t.C[A],t.C[A]=D,D=t.s[e],t.s[e]=t.s[A],t.s[A]=D,D=t.A[e],t.A[e]=t.A[A],t.A[A]=D,D=t.u[e],t.u[e]=t.u[A],t.u[A]=D,o=t.B[e],t.B[e]=t.B[A],t.B[A]=o,D=t.w[e],t.w[e]=t.w[A],t.w[A]=D,i=t.J[e],t.J[e]=t.J[A],t.J[A]=i,t.v!=null&&(u=t.v[e],t.v[e]=t.v[A],t.v[A]=u),t.t!=null&&(u=t.t[e],t.t[e]=t.t[A],t.t[A]=u),P=0;P<t.r;P++)for(r=0;r<2;r++)t.D[r][P]==e?t.D[r][P]=A:t.D[r][P]==A&&(t.D[r][P]=e);t.T=0;},n.Zn=function(t,e,A){var P;P=t.D[0][e],t.D[0][e]=t.D[0][A],t.D[0][A]=P,P=t.D[1][e],t.D[1][e]=t.D[1][A],t.D[1][A]=P,P=t.H[e],t.H[e]=t.H[A],t.H[A]=P,P=t.F[e],t.F[e]=t.F[A],t.F[A]=P,P=t.G[e],t.G[e]=t.G[A],t.G[A]=P,t.T=0;},n.$n=function(t,e,A){var P;for(P=0;P<t.q;P++)t.J[P].a+=e,t.J[P].b+=A;t.U+=e,t.V+=A;},n._n=function(t,e,A){var P,r;switch(r=A&127,P=n.pm(t,e),r){case 1:case 8:return P>=1;case 2:return P>=2;case 4:return P>=3;case 32:return P>=4;case 64:return P>=5;case 16:return  true;default:return  false;}},n.ao=function(t,e,A,P){var r,i,D;for(r=0;r<t.q;r++)(!P||(t.u[r]&512)!=0)&&(D=t.R[r]*e,i=t.Q[r]-A,t.J[r].a=t.U+D*j.Math.sin(i),t.J[r].b=t.V+D*j.Math.cos(i));P&&(t.T&=7);},n.bo=function(t,e,A){var P,r,i;for(t.U=e,t.V=A,t.Q=z(l.b7,jt,6,t.q,15,1),t.R=z(l.b7,jt,6,t.q,15,1),P=0;P<t.q;P++)r=e-t.J[P].a,i=A-t.J[P].b,t.R[P]=j.Math.sqrt(r*r+i*i),t.Q[P]=n.mo(e,A,t.J[P].a,t.J[P].b);},n.co=function(){n.Qk(),this.M=this.N=256,n.sm(this);},n.eo=function(t,e){this.M=j.Math.max(1,t),this.N=j.Math.max(1,e),n.sm(this);},n.go=function(t,e){var A;return A=z(l.d7,H,6,e,15,1),T.Vjb(t,0,A,0,j.Math.min(t.length,e)),A;},n.ho=function(t,e){var A;return A=z(l.e7,n0,6,e,14,1),T.Vjb(t,0,A,0,j.Math.min(t.length,e)),A;},n.io=function(t,e){var A,P;for(A=z(l.w7,Sn,29,e,0,1),P=0;P<t.length;P++)t[P]&&(A[P]=new n.oi(t[P]));return A;},n.jo=function(t,e){var A,P;for(A=z(l._6,Te,10,e,0,2),P=0;P<t.length;P++)t[P]!=null&&(A[P]=z(l._6,J,6,t[P].length,15,1),T.Vjb(t[P],0,A[P],0,t[P].length));return A;},n.ko=function(t,e){var A,P;for(A=z(l.d7,S1,7,e,0,2),P=0;P<t.length;P++)t[P]!=null&&(A[P]=z(l.d7,H,6,t[P].length,15,1),T.Vjb(t[P],0,A[P],0,t[P].length));return A;},n.lo=function(t){return n.Qk(),t>=0&&t<n.Lk.length&&n.Lk[t]!=null?n.Lk[t]:t>=171&&t<=190?n.Jk:n.Nk;},n.mo=function(t,e,A,P){n.Qk();var r,i,D;return i=A-t,D=P-e,D!=0?(r=j.Math.atan(i/D),D<0&&(i<0?r-=n1:r+=n1)):r=i>0?m0:uA,r;},n.no=function(t,e){n.Qk();var A;for(A=t-e;A<lr;)A+=X1;for(;A>n1;)A-=X1;return A;},n.oo=function(t,e){n.Qk();var A,P,r,i,D;if((e&256)!=0&&T.ejb(t,"?"))return 0;for(P=1;P<=128;P++)if(!T.ejb(t,R1)&&T.fjb(t,n.Kk[P]))return P;if((e&2)!=0){for(r=129;r<=144;r++)if(T.fjb(t,n.Kk[r]))return r;}if((e&4)!=0){for(i=146;i<=148;i++)if(T.fjb(t,n.Kk[i]))return i;}if((e&1)!=0){for(D=151;D<=152;D++)if(T.fjb(t,n.Kk[D]))return D;}if((e&32)!=0&&T.fjb(t,n.Kk[153]))return 153;if((e&8)!=0&&T.fjb(t,n.Kk[154]))return 154;if((e&16)!=0&&T.fjb(t,n.Kk[145]))return 145;if((e&128)!=0&&T.fjb(t,n.Kk[159]))return 159;if((e&64)!=0){for(A=171;A<=190;A++)if(T.fjb(t,n.Kk[A]))return A;}return 0;},n.po=function(t){switch(n.Qk(),t){case 7:case 8:case 9:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:return  true;}return  false;},n.qo=function(t){return n.Qk(),!(t==1||t==6||n.po(t)||t==2||t==10||t==18||t==36||t==54||t==86||t>103);},n.ro=function(t){return t>=3&&t<=4||t>=11&&t<=13||t>=19&&t<=31||t>=37&&t<=51||t>=55&&t<=84||t>=87&&t<=112;},n.so=function(t){return t==2||t==10||t==18||t==36||t==54||t==86;},n.to=function(t){return t==1||t>=5&&t<=9||t>=14&&t<=17||t>=32&&t<=35||t>=52&&t<=53;},n.uo=function(t){return t>=21&&t<=30||t>=39&&t<=48||t==57||t>=72&&t<=80||t==89||t>=104&&t<=112;},x(88,1,{88:1,4:1},n.co),y.Xb=function(t){n.ol(this,t);},y.Yb=function(t){return this.H[t]==8;},y.q=0,y.r=0,y.I=0,y.K=false,y.L=false,y.M=0,y.N=0,y.O=0,y.S=false,y.T=0,y.U=0,y.V=0,n.Pk=24,l.L7=X(88),n.wo=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h;for(n.aw(e,1),P==null&&(P=z(l.d7,H,6,e.q,15,1)),u=n.bn(t,1),o=n.bn(t,2),h=z(l.zeb,ht,6,e.q,16,1),Q=z(l.d7,H,6,e.q,15,1),Q[0]=A,h[A]=true,P[A]=n.kl(e,t,A,u,o),D=0,L=0;D<=L;){for(s=0;s<n.Ko(e,Q[D]);s++)i=e.i[Q[D]][s],h[i]||(Q[++L]=i,h[i]=true,P[i]=n.kl(e,t,i,u,o));++D;}for(r=0;r<e.r;r++)h[e.D[0][r]]&&n.ll(e,t,r,u,o,P==null?e.D[0][r]:P[e.D[0][r]],P==null?e.D[1][r]:P[e.D[1][r]],false);n.bn(t,1),n.bn(t,2),t.T=0;},n.xo=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h;for(t.j=z(l.d7,H,6,t.q,15,1),t.e=z(l.d7,H,6,t.q,15,1),t.i=z(l.d7,S1,7,t.q,0,2),t.k=z(l.d7,S1,7,t.q,0,2),t.n=z(l.d7,S1,7,t.q,0,2),t.o=z(l.d7,H,6,t.f,15,1),Q=z(l.d7,H,6,t.q,15,1),D=0;D<t.r;D++)++Q[t.D[0][D]],++Q[t.D[1][D]];for(P=0;P<t.q;P++)t.i[P]=z(l.d7,H,6,Q[P],15,1),t.k[P]=z(l.d7,H,6,Q[P],15,1),t.n[P]=z(l.d7,H,6,Q[P],15,1);for(s=false,u=0;u<t.g;u++){if(h=n.em(t,u),h==0){s=true;continue;}for(L=0;L<2;L++)A=t.D[L][u],e=t.e[A],t.n[A][e]=h,t.i[A][e]=t.D[1-L][u],t.k[A][e]=u,++t.e[A],++t.j[A],A<t.f&&(h>1?t.o[A]+=h-1:t.H[u]==8&&(t.o[A]=1));}for(o=t.g;o<t.r;o++){if(h=n.em(t,o),h==0){s=true;continue;}for(L=0;L<2;L++)A=t.D[L][o],e=t.e[A],t.n[A][e]=h,t.i[A][e]=t.D[1-L][o],t.k[A][e]=o,++t.e[A],t.D[1-L][o]<t.f&&++t.j[A];}if(s){for(e=z(l.d7,H,6,t.q,15,1),r=0;r<t.q;r++)e[r]=t.e[r];for(i=0;i<t.r;i++)if(h=n.em(t,i),h==0)for(L=0;L<2;L++)A=t.D[L][i],t.n[A][e[A]]=h,t.i[A][e[A]]=t.D[1-L][i],t.k[A][e[A]]=i,++e[A];}},n.yo=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V;for(n.aw(t,1),A&&(e=true),o=0;o<t.r;o++)if(Q=n.em(t,o),Q==1||Q==2){if(t.s[t.D[0][o]]>0&&t.s[t.D[1][o]]<0)i=t.D[0][o],D=t.D[1][o];else if(t.s[t.D[0][o]]<0&&t.s[t.D[1][o]]>0)i=t.D[1][o],D=t.D[0][o];else continue;if(n.Km(t,i)||n.Km(t,D)||t.C[i]<9&&n.up(t,i)>3||t.C[D]<9&&n.up(t,D)>3)continue;s=n.np(t,i)!=0,t.s[i]-=1,t.s[D]+=1,s||(C=t.H[o],Q==1?t.H[o]=2:t.H[o]=4,(C==129||C==257)&&(V=t.D[0][o],m=n.oq(t,V,false),t.D[0][m]!=V&&(t.D[1][m]=t.D[0][m],t.D[0][m]=V))),t.T=0;}for(v=0,g=0,b=0,r=0;r<t.q;r++)v+=t.s[r],t.s[r]<0&&!n.Op(t,r)&&(++g,n.Gm(t,r)&&(b-=t.s[r]));if(!e&&v!=0)throw Ut(new T.bC("molecule's overall charges are not balanced"));for(n.aw(t,1),M=0,S=A?v+b:b,u=0;u<t.q;u++)t.s[u]>0&&!n.Np(t,u)&&n.Gm(t,u)&&(L=j.Math.min(n.np(t,u),t.s[u]),L!=0&&S>=L&&(v-=L,M-=L,S-=L,t.s[u]-=L,t.T&=1));if(d=A?v:M,d<0){for(c=z(l.d7,H,6,g,15,1),g=0,i=0;i<t.q;i++)t.s[i]<0&&!n.Op(t,i)&&(c[g++]=(t.C[i]<<22)+i);for(w.qzb(c,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),h=c.length-1;d<0&&h>=c.length-g;h--)P=c[h]&W1,n.Gm(t,P)&&(L=j.Math.min(-d,-t.s[P]),v+=L,d+=L,t.s[P]+=L,t.T&=1);}return v;},n.zo=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b;for(A=0;A<t.q;A++)b=n.om(t,A),b+n.mm(t,A,b,t.C[A])-n.up(t,A)<=0&&!(t.s[A]==0&&(t.C[A]==5||(r=t.C[A],r==7||r==15||r==33)||(P=t.C[A],P==8||P==16||P==34||P==52||P==84)))&&(t.B[A]=G(t.B[A],-6145));if(t.S)return  false;for(D=false,e=0;e<t.f;e++)if(u=t.e[e]-t.j[e],u>0)for(M1(G(t.B[e],z1),0)&&(L=M1(G(t.B[e],Z0),896)?3:M1(G(t.B[e],Z0),384)?2:M1(G(t.B[e],128),128)?1:0,o=(b=n.om(t,e),b+n.mm(t,e,b,t.C[e])-n.up(t,e)),t.s[e]==0&&M1(G(t.B[e],je),0)&&t.C[e]!=6&&++o,s=u,s>3-L&&(s=3-L),s>o+u-L&&(s=o+u-L),s>0&&(h=L==0?0:b1(G(t.B[e],Z0),s),h=Pt(h,l.Oeb((s==3?7:u==2?3:1)<<7)),t.B[e]=G(t.B[e],-1921),t.B[e]=Pt(t.B[e],G(Z0,h)))),Q=t.j[e];Q<t.e[e];Q++)i=t.k[e][Q],t.H[i]==1&&(t.C[t.i[e][Q]]=-1,t.H[i]=512,D=true);return D&&n.jl(t),D;},n.Ao=function(t,e){var A,P,r,i;if(t.o[e]==2&&t.j[e]==2&&t.n[e][0]==2){for(r=0;r<2;r++)if(A=n.Fo(t,e,t.i[e][r]),A!=-1)for(i=0;i<t.j[A];i++)P=t.k[A][i],(t.H[P]==257||t.H[P]==129)&&t.D[0][P]==A&&(t.H[t.k[A][i]]=1);return;}if(t.o[e]==0||t.C[e]>=15)for(r=0;r<t.e[e];r++)P=t.k[e][r],(t.H[P]==257||t.H[P]==129)&&t.D[0][P]==e&&t.D[0][P]==e&&(t.H[P]=1);},n.Bo=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h;for(n.aw(t,P?7:1),e.v=null,t.K&&n.Nn(e,true),o=A.length,r==null&&(r=z(l.d7,H,6,o,15,1)),e.q=0,i=0;i<o;i++)if(A[i]){if(r[i]=n.kl(t,e,i,0,0),t.C[i]==1&&(h=n.up(t,i),h>1)){for(s=0;s<t.e[i];s++)if(!A[t.i[i][s]]){n.fn(e,r[i],h);break;}}}else r[i]=-1;for(L=z(l.d7,H,6,t.r,15,1),U.Oqb(L,L.length,-1),e.r=0,Q=0;Q<t.r;Q++)D=t.D[0][Q],u=t.D[1][Q],D<o&&u<o&&(A[D]&&A[u]?L[Q]=n.ll(t,e,Q,0,0,r==null?t.D[0][Q]:r[t.D[0][Q]],r==null?t.D[1][Q]:r[t.D[1][Q]],P):t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(A[D]&&(e.s[r[D]]+=t.s[D]<0?1:-1),A[u]&&(e.s[r[u]]+=t.s[u]<0?1:-1)));n.ol(t,e),t.d&&(e.T=0),e.T=0,n.bn(e,1),n.bn(e,2),e.q!=o&&n.Nn(e,true),n.rq(t,e,o,r,L),n.qq(t,e,o,r),P&&n.Fe(new n.Ue(e),null,false);},n.Co=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h;for(n.aw(t,P?7:1),e.v=null,t.K&&n.Nn(e,true),r==null&&(r=z(l.d7,H,6,t.q,15,1)),e.q=0,i=0;i<t.q;i++)for(r[i]=-1,L=0;L<t.j[i];L++)if(A[t.k[i][L]]){if(r[i]=n.kl(t,e,i,0,0),t.C[i]==1&&(h=n.up(t,i),h>1)){for(s=0;s<t.e[i];s++)if(!A[t.k[i][s]]){n.fn(e,r[i],h);break;}}break;}for(Q=z(l.d7,H,6,t.r,15,1),e.r=0,o=0;o<t.r;o++)A[o]?Q[o]=n.ll(t,e,o,0,0,r==null?t.D[0][o]:r[t.D[0][o]],r==null?t.D[1][o]:r[t.D[1][o]],P):(Q[o]=-1,D=t.D[0][o],u=t.D[1][o],r[D]==-1^r[u]==-1&&t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(r[D]!=-1&&(e.s[r[D]]+=t.s[D]<0?1:-1),r[u]!=-1&&(e.s[r[u]]+=t.s[u]<0?1:-1)));return n.ol(t,e),t.d&&(e.T=0),e.T=0,n.bn(e,1),n.bn(e,2),e.q!=t.q&&n.Nn(e,true),n.rq(t,e,t.q,r,Q),n.qq(t,e,t.q,r),P&&n.Fe(new n.Ue(e),null,false),r;},n.Do=function(t,e){var A,P,r,i,D,u,o,Q,L,s;if((e&~t.T)!=0&&((t.T&1)==0&&(n.Mp(t),n.xo(t),t.T|=1,t.K&&!n.um(t)&&n.zo(t)&&(n.Mp(t),n.xo(t),t.T|=1)),(e&~t.T)!=0)){if((t.T&-7)!=0){for(P=0;P<t.f;P++)t.u[P]&=-15369;for(i=0;i<t.g;i++)t.F[i]&=-705;if((e&4)==0){n.Io(t,1),t.T|=2;return;}for(n.Io(t,7),r=0;r<t.f;r++)for(L=0;L<t.j[r];L++)if(o=t.k[r][L],!(o<t.g&&n.ot(t.p,o)))for(u=t.i[r][L],s=0;s<t.j[u];s++)t.k[u][s]!=o&&t.n[u][s]>1&&(t.C[t.i[u][s]]==6?t.u[r]|=Y1:!n.Tp(t,t.k[u][s])&&n.Gm(t,t.i[u][s])&&(t.u[r]|=Q0));for(;;){for(Q=false,A=0;A<t.f;A++)if(t.o[A]>0&&(t.u[A]&Q0)!=0&&!n.nt(t.p,A)){for(L=0;L<t.j[A];L++)if(t.n[A][L]>1)for(u=t.i[A][L],o=t.k[A][L],s=0;s<t.j[u];s++)t.k[u][s]!=o&&(D=t.i[u][s],(t.u[D]&Q0)==0&&(t.u[D]|=Q0,Q=true));}if(!Q)break;}}t.T|=6;}},n.Eo=function(t,e){var A,P,r,i,D;if(A=-1,t.o[e]==1){for(i=0;i<t.j[e];i++)if(t.n[e][i]==2){if(P=t.i[e][i],t.j[P]==2&&t.o[P]==2){for(D=0;D<2;D++)if(r=t.i[P][D],r!=e&&t.o[r]==1){A=P;break;}}break;}}return A;},n.Fo=function(t,e,A){var P,r;for(P=e;t.j[A]==2&&t.o[A]==2&&A!=P;)r=A,A=t.i[A][0]==e?t.i[A][1]:t.i[A][0],e=r;return A==P?-1:A;},n.Go=function(t,e){var A;if(t.j[e]==3&&e<t.f&&n.nt(t.p,e)&&(t.p&&e<t.f?n.ct(t.p,e):0)>=5){for(A=0;A<t.j[e];A++)if(n.Up(t,t.k[e][A]))return t.k[e][A];}return  -1;},n.Ho=function(t,e,A,P,r){var i,D,u,o,Q,L;if(n.aw(t,7),!((t.u[e]&V0)==0||A&&!(e<t.f&&n.nt(t.p,e))))for(o=z(l.d7,H,6,t.f,15,1),o[0]=e,P[e]=true,u=0,Q=0;u<=Q;){for(L=0;L<t.j[o[u]];L++)D=t.k[o[u]][L],!r[D]&&(t.F[D]&64)!=0&&(!A||D<t.g&&n.ot(t.p,D))&&(r[D]=true,i=t.i[o[u]][L],P[i]||(P[i]=true,o[++Q]=i));++u;}},n.Io=function(t,e){var A,P,r,i,D,u,o,Q;for(t.p=new n.Ct(t,e),P=z(l.d7,H,6,t.f,15,1),r=0;r<t.g;r++)n.et(t.p,r)!=0&&(t.F[r]|=64,++P[t.D[0][r]],++P[t.D[1][r]]);for(A=0;A<t.f;A++)P[A]==2?t.u[A]|=i0:P[A]==3?t.u[A]|=z1:P[A]>3&&(t.u[A]|=V0);for(Q=0;Q<t.p.j.a.length;Q++)for(D=n.gt(t.p,Q),o=n.ht(t.p,Q),u=D.length,i=0;i<u;i++)t.u[D[i]]|=8,t.F[o[i]]|=128,t.H[o[i]]==386&&(t.H[o[i]]=2);},n.Jo=function(t){var e,A,P,r,i,D,u,o;for(u=z(l.zeb,ht,6,t.q,16,1),A=0;A<t.q;A++)u[A]=t.C[A]==1&&t.A[A]==0&&t.s[A]==0&&(t.u[A]&be)==0&&(t.t==null||t.t[A]==null);for(o=z(l.zeb,ht,6,t.q,16,1),D=0;D<t.r;D++){if(P=t.D[0][D],r=t.D[1][D],n.em(t,D)!=1){u[P]=false,u[r]=false;continue;}o[P]&&(u[P]=false),o[r]&&(u[r]=false),u[P]&&n.Km(t,r)&&t.C[r]!=13&&(u[P]=false),u[r]&&n.Km(t,P)&&t.C[P]!=13&&(u[r]=false),o[P]=true,o[r]=true;}for(i=0;i<t.r;i++)u[t.D[0][i]]&&u[t.D[1][i]]&&(u[t.D[0][i]]=false,u[t.D[1][i]]=false);for(e=0;e<t.q;e++)o[e]||(u[e]=false);return u;},n.Ko=function(t,e){return t.e[e];},n.Lo=function(t,e){return t.i[e].length;},n.Mo=function(t,e){return t.e[e]-n.sp(t,e)+n.np(t,e);},n.No=function(t){var e,A;for(n.aw(t,7),e=0,A=0;A<t.p.j.a.length;A++)n.mt(t.p,A)&&++e;return e;},n.Oo=function(t,e){var A,P;for(A=0,P=0;P<t.j[e];P++)n.Gm(t,t.i[e][P])&&!n.Bm(t,t.k[e][P])&&++A;return A;},n.Po=function(t,e){return t.o[e];},n.Qo=function(t,e){return n.aw(t,7),t.o[e]==2&&t.j[e]==2?n.mq(t,e,false):n.oq(t,e,false);},n.Ro=function(t,e){var A;return A=t.u[e]&V0,A==0?0:A==i0?2:A==z1?3:4;},n.So=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;for(n.aw(t,7),i=z(l.zeb,ht,6,t.g,16,1),s=z(l.zeb,ht,6,t.g,16,1),c=z(l.d7,H,6,t.f,15,1),D=0,u=1;u<t.j[e];u++)if(P=t.k[e][u],(t.F[P]&64)!=0){for(Q=0;Q<u;Q++)if(r=t.k[e][Q],(t.F[r]&64)!=0&&(s[P]=true,s[r]=true,b=n.xp(t,c,t.i[e][u],t.i[e][Q],A-2,null,s),s[P]=false,s[r]=false,b!=-1)){for(o=false,h=z(l.d7,H,6,b,15,1),n.yp(t,c,h,b),L=0;L<b;L++)i[h[L]]||(i[h[L]]=true,o=true);o&&++D;}}return D;},n.To=function(t,e){return t.p&&e<t.f?n.ct(t.p,e):0;},n.Uo=function(t,e){return e?(n.aw(t,1),n.Vl(t,t.f,t.g,n.Pk)):n.Vl(t,t.q,t.r,n.Pk);},n.Vo=function(t){var e,A,P,r,i,D,u,o;for(n.aw(t,1),u=z(l.c7,w0,6,t.f,15,1),P=z(l.d7,H,6,t.f,15,1),o=0;o<t.f;o++){for(P[0]=o,r=z(l.d7,H,6,t.f,15,1),r[o]=1,A=0,i=0;A<=i;){for(D=0;D<t.j[P[A]];D++)e=t.i[P[A]][D],r[e]==0&&(r[e]=r[P[A]]+1,P[++i]=e,u[o]+=r[e]-1);++A;}u[o]/=i;}return u;},n.Wo=function(t,e,A){var P;for(P=0;P<t.i[e].length;P++)if(t.i[e][P]==A)return t.k[e][P];return  -1;},n.Xo=function(t,e){return t.p&&e<t.g?n.et(t.p,e):0;},n.Yo=function(t,e,A){return t.i[e][A];},n.Zo=function(t,e){return t.j[e];},n.$o=function(t,e,A){return t.k[e][A];},n._o=function(t,e,A){return t.n[e][A];},n.ap=function(t,e){var A,P;if(A=0,t.K)for(P=0;P<t.j[e];P++)n.Im(t,t.i[e][P])&&++A;return A;},n.bp=function(t,e){return t.e[e]-t.j[e];},n.cp=function(t,e,A,P,r){var i,D,u,o;if(D=t.e[e],t.o[e]!=0||e<t.f&&n.nt(t.p,e)||t.j[e]<3||D>4)return  false;for(o=z(l.zeb,ht,6,4,16,1),u=0;u<D;u++){if(i=3.9269908169872414-P[u],j.Math.abs(Qr-i%m0)>.0872664675116539||(r[u]=3&Ft(i/m0),o[r[u]]))return  false;if(o[r[u]]=true,(r[u]&1)==0){if(t.H[t.k[e][A[u]]]!=1)return  false;}else if(!n.Rm(t,t.k[e][A[u]],e))return  false;}return o[0]&&o[2];},n.dp=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h;if((t.p&&e<t.f?n.ct(t.p,e):0)>24||(i=t.e[e],r==null&&(r=z(l.d7,H,6,i,15,1)),!n.cp(t,e,A,P,r)))return 3;for(o=-1,Q=0;Q<i;Q++)if((r[Q]&1)==1){if(D=t.H[t.k[e][A[Q]]],o!=-1&&o!=D)return 3;o=D;}return L=j.Math.abs(r[0]-r[1])==2?1:0,u=r[L]-r[L+1],h=j.Math.abs(u)==3^r[L]<r[L+1],s=i==3||(r[3]&1)==1,h^s^o==129?1:2;},n.ep=function(t,e,A){var P,r,i,D;for(D=D=z(l.zeb,ht,6,t.q,16,1),i=n.fp(t,e,A,D),r=z(l.d7,H,6,i,15,1),i=0,P=0;P<t.q;P++)D[P]&&(r[i++]=P);return r;},n.fp=function(t,e,A,P){var r,i,D,u,o,Q,L;for(n.aw(t,1),P==null&&(P=z(l.zeb,ht,6,t.q,16,1)),o=z(l.d7,H,6,t.q,15,1),o[0]=e,P[e]=true,D=0,Q=0,u=1;D<=Q;){for(i=A?n.Lo(t,o[D]):t.e[o[D]],L=0;L<i;L++)r=t.i[o[D]][L],r<P.length&&!P[r]&&(o[++Q]=r,P[r]=true,++u);++D;}return u;},n.gp=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h;for(n.aw(t,1),i=0;i<t.q;i++)e[i]=-1;for(Q=0,r=0;r<t.q;r++)if(e[r]==-1&&(!A||(t.u[r]&A0)!=0)){for(e[r]=Q,L=z(l.d7,H,6,t.q,15,1),L[0]=r,o=0,s=0;o<=s;){for(u=P?n.Lo(t,L[o]):t.e[L[o]],h=0;h<u;h++)D=t.i[L[o]][h],e[D]==-1&&(!A||(t.u[D]&A0)!=0)&&(L[++s]=D,e[D]=Q);++o;}++Q;}return Q;},n.hp=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h;for(n.aw(t,1),i=0;i<t.q;i++)e[i]=-1;for(Q=0,r=0;r<t.q;r++)if(e[r]==-1){for(e[r]=Q,L=z(l.d7,H,6,t.q,15,1),L[0]=r,o=0,s=0;o<=s;){for(u=P?n.Lo(t,L[o]):t.e[L[o]],h=0;h<u;h++)D=t.i[L[o]][h],e[D]==-1&&!A[t.k[L[o]][h]]&&(L[++s]=D,e[D]=Q);++o;}++Q;}return Q;},n.ip=function(t,e){var A;return A=n.om(t,e),A+n.mm(t,e,A,t.C[e])-n.up(t,e);},n.jp=function(t,e){var A,P,r,i,D,u;for(i=z(l.d7,H,6,t.q,15,1),P=0;P<t.q;P++)i[P]=P;r=t.q;do--r;while(r>=0&&e[r]);for(A=0;A<r;A++)if(e[i[A]]){u=i[A],i[A]=i[r],i[r]=u,D=e[A],e[A]=e[r],e[r]=D;do--r;while(e[r]);}return i;},n.kp=function(t){return n.jp(t,n.Jo(t));},n.lp=function(t,e,A){var P,r,i;if(r=n.up(t,e),r-=n.mm(t,e,r,t.C[e]),A&&(r-=t.e[e]-t.j[e]),i=n.lo(t.C[e]),r<=i[0])return  -1;for(P=1;P<i.length;P++)if(i[P]>=r)return i[P];return r;},n.mp=function(t){var e,A;if(t.K)return 0;for(n.aw(t,1),A=0,e=0;e<t.f;e++)A+=n.np(t,e);return A;},n.np=function(t,e){var A,P,r,i,D,u,o;if(t.K&&M1(G(t.B[e],z1),0)||!n.zq(t,e))return 0;if(t.C[e]==0||T.ejb("*",t.t==null||t.t[e]==null?null:T.cjb((i=t.t[e],V1.Hmb(),i))))return t.e[e]==0?1:0;for(n.aw(t,1),u=0,r=0;r<t.e[e];r++)u+=t.n[e][r];if(t.K){for(A=1,P=0;P<t.j[e];P++)t.H[t.k[e][P]]==8&&++A;u+=A>>1;}if(u-=n.mm(t,e,u,t.C[e]),D=((t.u[e]&be)>>>27)-1,D==-1)for(o=n.lo(t.C[e]),D=o[0],P=1;D<u&&P<o.length;P++)D=o[P];return j.Math.max(0,D-u);},n.op=function(t,e){var A,P,r,i,D;if(!t.K||t.v==null||t.v[e]==null)return n.pp(t,e,t.C[e]);for(D=0,P=t.v[e],r=0,i=P.length;r<i;++r)A=P[r],D=j.Math.max(D,n.pp(t,e,A));return D;},n.pp=function(t,e,A){var P,r,i,D,u;if(i=n.up(t,e),P=n.mm(t,e,i,A),D=((t.u[e]&be)>>>27)-1,D==-1){for(u=n.lo(t.C[e]),r=0;r<u.length-1&&i>u[r]+P;)++r;D=u[r];}return D+P-i;},n.qp=function(t,e){return t.i[e].length-t.e[e];},n.rp=function(t){var e,A,P,r;for(n.aw(t,1),r=0,e=0;e<t.q;e++)P=t.A[e]!=0?t.A[e]:n.Ok[t.C[e]],r+=P+n.np(t,e)*n.Ok[1],t.C[e]>=171&&t.C[e]<=190&&(A=t.e[e],A>2&&(r-=(A-2)*n.Ok[1]));return r;},n.sp=function(t,e){var A,P;for(A=t.j[e],P=0;P<t.j[e];P++)t.C[t.i[e][P]]==1&&--A;return A;},n.tp=function(t,e){return t.j[e]-n.ap(t,e);},n.up=function(t,e){var A,P,r,i,D,u;for(t.Zb(1),D=false,P=false,u=0,r=0;r<t.e[e];r++)(!t.K||!n.Im(t,t.i[e][r]))&&(i=t.n[e][r],u+=i,i>1&&(D=true),A=t.k[e][r],t.H[A]==8&&(P=true));return P&&!D&&++u,u;},n.vp=function(t,e,A){var P,r,i;for(r=0,i=0;i<t.j[e];i++)P=t.i[e][i],P!=A&&t.j[P]>2&&++r;return r;},n.wp=function(t,e){var A,P,r,i,D,u,o,Q,L,s;if(n.so(t.C[e]))return 0;if(t.C[e]==1)return t.j[e]==0?0:n.qo(t.C[t.i[e][0]])?-1:1;if(s=t.s[e],o=t.e[e]-n.sp(t,e)+n.np(t,e),t.C[e]!=1&&t.C[e]!=15&&(n.qo(t.C[e])?s+=o:s-=o),t.j[e]==0)return s;for(L=0;L<t.j[e];L++)i=t.i[e][L],A=n.Xp(t,t.k[e][L])?1:t.n[e][L],n.qo(t.C[e])?n.qo(t.C[i])||(s+=A):n.qo(t.C[i])?s-=A:t.C[e]!=t.C[i]&&(n.Ws(t.C[e]).a<n.Ws(t.C[i]).a?s+=A:s-=A);if(e<t.f&&n.st(t.p,e)){for(P=0,r=0,u=0,Q=0;Q<t.j[e];Q++)n.Xp(t,t.k[e][Q])&&(D=t.C[t.i[e][Q]],D==6?++r:n.po(D)?++u:++P);t.C[e]==6?P>1?--s:u>1&&++s:n.Gm(t,e)?P+r>1&&--s:u+r>1&&++s;}return s;},n.xp=function(t,e,A,P,r,i,D){var u,o,Q,L,s,h,b,c,g;if(A==P)return e[0]=A,0;for(t.Zb(1),L=z(l.d7,H,6,t.q,15,1),Q=z(l.d7,H,6,t.q,15,1),g=z(l.d7,H,6,t.q,15,1),Q[0]=A,L[A]=1,o=0,s=0;o<=s&&L[Q[o]]<=r;){for(c=Q[o],h=0;h<t.e[c];h++)if(D==null||D.length<=t.k[c][h]||!D[t.k[c][h]]){if(u=t.i[c][h],u==P){for(b=L[c],e[b]=u,e[--b]=c;b>0;)e[b-1]=g[e[b]],--b;return L[c];}L[u]==0&&(i==null||i.length<=u||!i[u])&&(Q[++s]=u,L[u]=L[c]+1,g[u]=c);}++o;}return  -1;},n.yp=function(t,e,A,P){var r,i;for(n.aw(t,1),r=0;r<P;r++)for(i=0;i<t.e[e[r]];i++)if(t.i[e[r]][i]==e[r+1]){A[r]=t.k[e[r]][i];break;}},n.zp=function(t,e,A){var P,r,i,D,u,o;if(e==A)return 0;for(n.aw(t,1),D=z(l.d7,H,6,t.q,15,1),i=z(l.d7,H,6,t.q,15,1),i[0]=e,D[e]=1,r=0,u=0;r<=u;){for(o=0;o<t.e[i[r]];o++){if(P=t.i[i[r]][o],P==A)return D[i[r]];D[P]==0&&(i[++u]=P,D[P]=D[i[r]]+1);}++r;}return  -1;},n.Ap=function(t,e,A,P,r){var i,D,u,o,Q,L;if(e==A)return 0;for(t.Zb(1),o=z(l.d7,H,6,t.q,15,1),u=z(l.d7,H,6,t.q,15,1),u[0]=e,o[e]=1,D=0,Q=0;D<=Q&&o[u[D]]<=P;){for(L=0;L<t.e[u[D]];L++){if(i=t.i[u[D]][L],i==A)return o[u[D]];o[i]==0&&(r==null||r.length<=i||!r[i])&&(u[++Q]=i,o[i]=o[u[D]]+1);}++D;}return  -1;},n.Bp=function(t,e){return t.e[e]-t.j[e]+n.np(t,e);},n.Cp=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c;for(c=z(l.d7,H,6,16,15,1),A=z(l.b7,jt,6,16,15,1),i=z(l.b7,jt,6,2,15,1),P=0,L=0;L<2;L++)for(r=t.D[L][e],s=0;s<t.j[r];s++)if(o=t.k[r][s],o!=e){if(P==4)return 0;u=t.i[r][s],c[P]=16,t.o[u]!=0&&(c[P]+=(t.u[u]&V0)!=0?1:4),(t.F[e]&64)!=0&&(t.F[o]&64)!=0&&(h=n.kt(t.p,e,o),h!=-1&&(c[P]+=n.mt(t.p,h)?64:6)),A[P++]=n.mo(t.J[r].a,t.J[r].b,t.J[u].a,t.J[u].b);}for(i[0]=n.Xl(t,t.D[0][e],t.D[1][e]),i[0]<0?(i[1]=i[0]+n1,D=false):(i[1]=i[0],i[0]=i[1]-n1,D=true),b=0,Q=0;Q<P;Q++)A[Q]>i[0]&&A[Q]<i[1]?b-=c[Q]:b+=c[Q];return D?-b:b;},n.Dp=function(t){return n.aw(t,7),t.p;},n.Ep=function(t){var e,A,P,r,i,D,u,o,Q;for(Q=0,n.aw(t,7),P=0;P<t.g;P++)if(n.em(t,P)==1&&(t.F[P]&64)==0){for(u=true,D=0;D<2;D++){if(e=t.D[D][P],t.j[e]==1){u=false;break;}if(t.C[e]==7&&!(e<t.f&&n.nt(t.p,e))){for(A=t.D[1-D][P],o=0;o<t.j[A];o++)if(r=t.i[A][o],i=t.k[A][o],i!=P&&n.em(t,i)>1&&!(r<t.f&&n.nt(t.p,r))&&n.Gm(t,r)){u=false;break;}}}u&&!n.bq(t,P)&&++Q;}return Q;},n.Fp=function(t,e,A){var P,r,i,D,u,o;for(r=0,i=0;i<t.j[e];i++)if(P=t.i[e][i],P!=A){for(D=0,u=0;u<t.j[P];u++)o=t.i[P][u],o!=e&&n.Tp(t,t.k[P][u])&&t.j[o]>2&&++D;D==2&&++r;}return r;},n.Gp=function(t,e){var A,P,r,i;for(A=t.e[e],i=z(l.d7,H,6,A,15,1),r=0;r<A;r++)i[r]=(t.i[e][r]<<16)+r;for(w.qzb(i,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),P=0;P<A;P++)i[P]&=r1;return i;},n.Hp=function(t,e){var A,P;if(n.aw(t,1),t.j[e]==2&&t.n[e][0]==2&&t.n[e][1]==2){for(A=0;A<2;A++)for(P=0;P<t.e[t.i[e][A]];P++)if(n.Rm(t,t.k[t.i[e][A]][P],t.i[e][A]))return t.k[t.i[e][A]][P];}else for(A=0;A<t.e[e];A++)if(n.Rm(t,t.k[e][A],e))return t.k[e][A];return  -1;},n.Ip=function(t,e,A){return n.em(t,e)!=1?0:16-t.e[A]+16*j.Math.max(0,j.Math.min(9,t.p&&e<t.g?n.et(t.p,e):0)-2)+((t.H[e]&384)==0||t.D[0][e]!=A?Oe:0)+((t.u[A]&3)==0?Y1:0)+(t.C[A]==1?z1:0)+(t.e[A]==1?i0:0)+((t.F[e]&64)!=0?0:512)+(t.C[A]!=6?256:0);},n.Jp=function(t,e,A,P,r,i){var D,u,o,Q,L;for(n.aw(t,1),r&&(n.il(r),r.d=null,r.K=false),o=z(l.d7,H,6,t.q,15,1),P==null?P=z(l.zeb,ht,6,t.q,16,1):U.Sqb(P,P.length),o[0]=e,o[1]=A,P[e]=true,P[A]=true,u=1,Q=1;u<=Q;){for(L=0;L<t.e[o[u]];L++){if(D=t.i[o[u]][L],D==e&&u!=1)return  -1;P[D]||(P[D]=true,o[++Q]=D);}++u;}return r&&(i==null&&(i=z(l.d7,H,6,P.length,15,1)),n.Bo(t,r,P,false,i),n.el(r,i[e],0,0,-1,0)),P[e]=false,Q;},n.Kp=function(t,e,A){var P,r,i,D,u,o;for(n.aw(t,1),i=z(l.d7,H,6,t.f,15,1),o=z(l.zeb,ht,6,t.f,16,1),i[0]=e,i[1]=A,o[e]=true,o[A]=true,r=1,D=1;r<=D;){for(u=0;u<t.j[i[r]];u++){if(P=t.i[i[r]][u],P==e&&r!=1)return  -1;o[P]||(o[P]=true,i[++D]=P);}++r;}return D;},n.Lp=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b;if(n.em(t,A)!=2&&!(A<t.g&&n.ot(t.p,A))||(b=t.F[A]&3,b!=1&&b!=2))return  -1;for(o=0;o<2;o++){for(P=t.D[o][A],r=t.D[1-o][A],h=-1,D=false,L=0;L<t.j[P];L++)i=t.i[P][L],i!=r&&(i==e?D=true:h=i);if(D){for(s=-1,u=-1,Q=0;Q<t.j[r];Q++)i=t.i[r][Q],i!=P&&(s==-1?s=i:i>s?u=i:(u=s,s=i));return t.j[P]==2?t.j[r]==2?b==2?s:-1:b==2?s:u:t.j[r]==2?b==2^e<h?-1:s:b==2^e<h?u:s;}}return  -1;},n.Mp=function(t){var e,A,P,r,i,D,u,o,Q,L;D=n.Jo(t),o=t.q;do--o;while(o>=0&&D[o]);for(e=0;e<o;e++)if(D[e]){n.Yn(t,e,o),L=D[e],D[e]=D[o],D[o]=L;do--o;while(D[o]);}if(t.f=o+1,t.q==t.f){t.g=t.r;return;}for(u=z(l.zeb,ht,6,t.r,16,1),i=0;i<t.r;i++)A=t.D[0][i],P=t.D[1][i],(D[A]||D[P])&&(u[i]=true);Q=t.r;do--Q;while(Q>=0&&u[Q]);for(r=0;r<Q;r++)if(u[r]){n.Zn(t,r,Q),u[r]=false;do--Q;while(u[Q]);}t.g=Q+1;},n.Np=function(t,e){var A;for(A=0;A<t.j[e];A++)if(t.s[t.i[e][A]]<0)return  true;return  false;},n.Op=function(t,e){var A;for(A=0;A<t.j[e];A++)if(t.s[t.i[e][A]]>0)return  true;return  false;},n.Pp=function(t,e){var A;return A=t.C[e],A==3||A==11||A==19||A==37||A==55||A==87;},n.Qp=function(t,e){return (t.u[e]&Y1)!=0;},n.Rp=function(t,e){var A,P,r,i,D,u;for(n.aw(t,1),D=0;D<2;D++)if(A=t.D[D][e],t.C[A]==7){for(P=t.D[1-D][e],u=0;u<t.j[P];u++)if(r=t.i[P][u],i=t.k[P][u],(t.C[r]==7||t.C[r]==8||t.C[r]==16)&&n.em(t,i)>=2)return  true;}return  false;},n.Sp=function(t,e){return e<t.f&&n.nt(t.p,e);},n.Tp=function(t,e){return e<t.g&&n.ot(t.p,e);},n.Up=function(t,e){var A,P,r,i,D,u;return t.H[e]!=1||e<t.g&&n.ot(t.p,e)||(t.F[e]&64)!=0&&(t.p&&e<t.g?n.et(t.p,e):0)<7||(A=t.D[0][e],!(A<t.f&&n.nt(t.p,A))||(t.p&&A<t.f?n.ct(t.p,A):0)<5)||(P=t.D[1][e],!(P<t.f&&n.nt(t.p,P))||(t.p&&P<t.f?n.ct(t.p,P):0)<5)?false:(r=n.vp(t,A,P),i=n.vp(t,P,A),(t.p&&A<t.f?n.ct(t.p,A):0)>5&&(t.p&&P<t.f?n.ct(t.p,P):0)>5?r+i>2:(D=n.Fp(t,A,P),u=n.Fp(t,P,A),r==2&&u>=1||i==2&&D>=1||D==2&&(i>=1||u>=1)||u==2&&(r>=1||D>=1)));},n.Vp=function(t,e){return t.o[e]==2&&t.j[e]==2&&t.n[e][0]==2&&t.n[e][1]==2&&t.C[e]<=7;},n.Wp=function(t,e){var A;return A=t.C[e],A==8||A==16||A==34||A==52||A==84;},n.Xp=function(t,e){return e<t.g&&(n.tt(t.p,e)||t.H[e]==8);},n.Yp=function(t,e){var A;return A=t.C[e],A==4||A==12||A==20||A==38||A==56||A==88;},n.Zp=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g;if(t.C[e]!=7||t.j[e]==4)return  false;if(e<t.f&&n.nt(t.p,e)||t.o[e]!=0||kt(G(t.B[e],Ie),0))return  true;if(t.s[e]==1)return  false;for(o=0;o<t.j[e];o++)if(n.Xo(t,t.k[e][o])==3)return  false;for(D=0,Q=0;Q<t.j[e];Q++)t.n[e][Q]==1&&(P=t.C[t.i[e][Q]],(P==8||P==9||P==17)&&++D);if(D==0){for(u=0;u<t.j[e];u++)if(r=t.i[e][u],t.o[r]!=0){if(r<t.f&&n.nt(t.p,r)){if((t.p&&r<t.f?n.ct(t.p,r):0)>=5){for(g=0,h=0;h<t.j[r];h++)c=t.i[r][h],c!=e&&n.sp(t,c)>=3&&++g;if(b=n.sp(t,e),g==2&&b>=2||g==1&&b==3)continue;}return !A||!n.dq(t,e);}for(s=0;s<t.j[r];s++)if(t.n[r][s]==2||n.Tp(t,t.k[r][s]))return !A||!n.dq(t,e);}}if(D<2)for(u=0;u<t.j[e];u++){for(r=t.i[e][u],L=false,i=false,s=0;s<t.j[r];s++)t.i[r][s]!=e&&(t.n[r][s]!=1&&(t.C[t.i[r][s]]==7||t.C[t.i[r][s]]==8||t.C[t.i[r][s]]==16)&&(L=true),t.n[r][s]==1&&t.C[t.i[r][s]]==7&&(i=true));if(L&&(!i||D==0))return !A||!n.dq(t,e);}return  false;},n.$p=function(t,e){var A;return A=t.C[e],A==9||A==17||A==35||A==53||A==85;},n._p=function(t,e){return e<t.f&&n.ut(t.p,e);},n.aq=function(t,e){var A;return A=t.C[e],A==7||A==15||A==33;},n.bq=function(t,e){var A,P,r,i,D,u;if(n.em(t,e)!=1)return  false;for(i=0;i<2;i++){for(A=t.D[i][e],u=t.D[1-i][e];t.o[A]==2&&t.j[A]==2&&t.C[A]<10;)for(D=0;D<2;D++)if(P=t.i[A][D],P!=u){if(t.j[P]==1||(r=t.k[A][D],n.em(t,r)==1&&r<e))return  true;u=A,A=P;break;}if(t.j[A]==1)return  true;}return  false;},n.cq=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c,g,d,m,C;if(h=z(l.zeb,ht,6,t.g,16,1),h[r]=true,b=z(l.d7,H,6,11,15,1),c=n.xp(t,b,P,e,10,null,h),c==-1)return  false;for(D=1;!n.pt(t.p,A,b[D]);)++D;if(i=c-D,u=b[D],C=n.jt(t.p,A),C==6&&i==2&&D==3&&n.Ro(t,b[1])>=3){for(m=n.gt(t.p,A),s=0;s<6;s++)if(e==m[s]){if(d=n.Bt(t.p,A,u==m[n.Bt(t.p,A,s+2)]?s-2:s+2),g=m[d],L=t.u[g]&V0,(L==0?0:L==i0?2:L==z1?3:4)>=3&&n.Ap(t,b[1],g,2,null)==2)return  true;break;}}if(o=t.o[u]==1||u<t.f&&n.nt(t.p,u)||n.Zp(t,u,false),Q=!o&&t.C[u]==7&&t.s[u]!=1,i==1)return !o&&!Q&&C<=4&&D<=3;switch(C){case 4:if(!o&&!Q&&D<=4)return  true;break;case 5:if(Q){if(D<=3)return  true;}else if(!o&&D<=4)return  true;break;case 6:if(i==2){if(o){if(D<=4)return  true;}else if(!Q&&D<=3)return  true;}else if(i==3){if(o){if(D<=6)return  true;}else if(D<=4)return  true;}break;case 7:if(i==3&&D<=3)return  true;}return  false;},n.dq=function(t,e){var A,P,r,i,D;if(e<t.f&&n.nt(t.p,e)||t.o[e]!=0||kt(G(t.B[e],Ie),0)||(P=t.u[e]&V0,(P==0?0:P==i0?2:P==z1?3:4)<3)||(D=t.p&&e<t.f?n.ct(t.p,e):0,D>7))return  false;for(i=0;i<t.p.j.a.length&&!(n.jt(t.p,i)==D&&n.pt(t.p,i,e));)++i;if(i>=i0&&i==t.p.j.a.length)return  false;for(r=0;r<t.j[e];r++)if(A=t.k[e][r],!n.qt(t.p,i,A)&&n.cq(t,e,i,t.i[e][r],A))return  true;return  false;},n.eq=function(t,e){return (t.u[e]&V0)!=0;},n.fq=function(t,e){return (t.F[e]&64)!=0;},n.gq=function(t,e){return t.C[e]==1&&t.A[e]==0&&t.s[e]==0&&(t.u[e]&be)==0&&(t.t==null||t.t[e]==null);},n.hq=function(t,e){return (t.u[e]&8)!=0;},n.iq=function(t,e){return (t.F[e]&128)!=0;},n.jq=function(t,e){return (t.u[e]&Q0)!=0;},n.kq=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(n.aw(t,1),n.lq(t),Q=false,A=0;A<t.f;A++)if(t.C[A]==7&&t.s[A]==0){if(s=n.up(t,A),s==4){for(L=0;L<t.j[A];L++)if(u=t.i[A][L],t.n[A][L]==1&&t.C[u]==8&&t.j[u]==1&&t.s[u]==0){Q=true,++t.s[A],--t.s[u];break;}}else if(s==5)for(L=0;L<t.j[A];L++){if(u=t.i[A][L],o=t.k[A][L],t.n[A][L]==2&&t.C[u]==8){Q=true,++t.s[A],--t.s[u],t.H[o]=1;break;}if(t.n[A][L]==3&&t.C[u]==7){Q=true,++t.s[A],--t.s[u],t.H[o]=2;break;}}}for(D=false,i=0;i<t.g;i++)for(L=0;L<2;L++)if(n.Gm(t,t.D[L][i])){e=t.D[1-L][i],r=t.C[e],(r==3||r==11||r==19||r==37||r==55||r==87||(P=t.C[e],P==4||P==12||P==20||P==38||P==56||P==88))&&(n.em(t,i)==1?(++t.s[e],--t.s[t.D[L][i]],t.H[i]=512,D=true):t.H[i]==16&&(t.H[i]=512,D=true));break;}return D&&(n.jl(t),Q=true),Q&&(t.T=0),Q;},n.lq=function(t){var e;for(e=0;e<t.g;e++)if(t.H[e]==8)return n.Fe(new n.Ue(t),null,false);return  false;},n.mq=function(t,e,A){var P,r,i,D,u,o,Q,L;for(Q=-1,r=0,u=0;u<2;u++)for(P=t.i[e][u],o=0;o<t.e[P];o++)i=t.i[P][o],i!=e&&(D=t.k[P][o],L=n.Ip(t,D,i),r<L&&(!A||!(t.H[D]==257||t.H[D]==129))&&(r=L,Q=D));return Q;},n.nq=function(t,e){var A,P,r,i,D,u,o,Q;for(o=-1,P=0,D=0;D<2;D++)for(A=t.D[D][e],u=0;u<t.e[A];u++)r=t.i[A][u],r!=t.D[1-D][e]&&(i=t.k[A][u],Q=n.Ip(t,i,r),P<Q&&(P=Q,o=i));return o;},n.oq=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;for(P=t.e[e],r=z(l.b7,jt,6,P,15,1),b=0;b<P;b++)r[b]=n.Xl(t,e,t.i[e][b]);for(c=1;c<P;c++)for(g=0;g<c;g++)if(i=j.Math.abs(n.no(r[c],r[g])),i<.08||i>G8){for(D=0,u=0,d=0;d<P;d++)d!=c&&d!=g&&(D+=j.Math.abs(lt.B3(r[c],r[d])),u+=j.Math.abs(lt.B3(r[g],r[d])));if(Q=D<u?t.k[e][c]:t.k[e][g],n.em(t,Q)==1&&(!A||!(t.H[Q]==257||t.H[Q]==129)))return Q;}for(m=-1,o=0,h=0;h<P;h++)L=t.i[e][h],s=t.k[e][h],C=n.Ip(t,s,L),o<C&&(!A||!(t.H[s]==257||t.H[s]==129))&&(o=C,m=s);return m;},n.pq=function(t,e){var A,P,r,i,D,u;for(n.aw(t,e?15:1),t.q=t.f,t.r=t.g,u=z(l.zeb,ht,6,t.f,16,1),r=0;r<t.f;r++)t.e[r]!=t.j[r]&&(A=n.lp(t,r,false),u[r]=t.e[r]==t.j[r]+1&&n.Qm(t,t.k[r][t.j[r]]),t.e[r]=t.j[r],A!=-1&&(D=n.lp(t,r,true),A!=D&&(i=((t.u[r]&be)>>>27)-1,(i==-1||i<A)&&n.fn(t,r,A))));if(e)for(P=0;P<t.f;P++)u[P]&&n.vq(t,P);t.T=0;},n.qq=function(t,e,A,P){var r,i,D,u;for(n.aw(e,1),i=0;i<A;i++)P[i]!=-1&&t.C[i]!=1&&t.e[i]!=n.Ko(e,P[i])&&(r=n.lp(t,i,false),r!=-1&&(u=n.lp(e,P[i],false),r!=u&&(D=n.yl(e,P[i]),(D==-1||D<r)&&n.fn(e,P[i],r))));},n.rq=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c,g;for(u=0;u<A;u++)if(P[u]!=-1&&t.e[u]>3&&(t.u[u]&q0)!=0){for(g=0,h=-1,s=-1,L=0;L<t.e[u];L++)r[t.k[u][L]]!=-1?++g:t.n[u][L]==1&&n.Qm(t,t.k[u][L])&&t.D[0][t.k[u][L]]==u&&(h=t.k[u][L],s=t.i[u][L]);if(h!=-1&&g>=3){for(i=n.mo(t.J[u].a,t.J[u].b,t.J[s].a,t.J[s].b),b=10,c=-1,Q=0;Q<t.e[u];Q++)t.n[u][Q]==1&&(!n.Qm(t,t.k[u][Q])||t.D[0][t.k[u][Q]]==u)&&r[t.k[u][Q]]!=-1&&(D=j.Math.abs(n.no(i,n.Xl(t,u,t.i[u][Q]))),b>D&&(b=D,c=t.k[u][Q]));c!=-1&&(o=r[c],n.Ln(e,o,t.H[c]==257?129:257),t.D[0][c]!=u&&(n.Cn(e,1,o,P[t.D[0][c]]),n.Cn(e,0,o,P[u])));}}},n.sq=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O;if(t.j[e]!=2||t.n[e][0]!=2||t.n[e][1]!=2||t.j[t.i[e][0]]<2||t.j[t.i[e][1]]<2||t.o[t.i[e][0]]!=1||t.o[t.i[e][1]]!=1){n.tn(t,e,0,false);return;}for(S=-1,M=-1,v=-1,d=-1,i=0,s=0;s<2;s++)for(P=t.i[e][s],c=0;c<t.e[P];c++)D=t.i[P][c],D!=e&&(u=t.k[P][c],V=n.Ip(t,u,D),i<V&&(i=V,M=D,S=u,v=P,d=t.i[e][1-s]));if(M!=-1){for(h=0;h<2;h++)for(P=t.i[e][h],c=0;c<t.e[P];c++)D=t.i[P][c],u=t.k[P][c],D!=e&&t.D[0][u]==P&&(t.H[u]=1);for(t.D[1][S]!=M&&(t.D[0][S]=t.D[1][S],t.D[1][S]=M),o=St,b=0;b<t.j[v];b++)D=t.i[v][b],D!=e&&o>D&&(o=D);for(m=z(l.d7,H,6,2,15,1),C=0,L=0;L<t.j[d];L++)D=t.i[d][L],D!=e&&(m[C++]=D);A=n.mo(t.J[e].a,t.J[e].b,t.J[d].a,t.J[d].b),r=0,C==2?(m[0]>m[1]&&(O=m[0],m[0]=m[1],m[1]=O),Q=n.no(A,n.Xl(t,d,m[0])),g=n.no(A,n.Xl(t,d,m[1])),r=Q-g):r=n.no(A,n.Xl(t,d,m[0])),r<0^(t.u[e]&3)==1^o==M?t.H[S]=257:t.H[S]=129;}},n.tq=function(t,e,A,P){var r,i,D,u,o,Q;if(r=t.e[e],u=z(l.d7,H,6,r,15,1),Q=n.dp(t,e,A,P,u),Q==3)return  false;for(D=(t.u[e]&3)==Q?257:129,o=0;o<r;o++)(u[o]&1)==1&&(i=t.k[e][A[o]],t.H[i]=D,t.D[0][i]!=e&&(t.D[1][i]=t.D[0][i],t.D[0][i]=e));return  true;},n.uq=function(t,e){t.T|=248&(8|e);},n.vq=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d;if(n.Ao(t,e),!((t.u[e]&3)==0||(t.u[e]&3)==3)){if(t.o[e]==2&&t.j[e]==2){n.sq(t,e);return;}if(t.j[e]<3||t.j[e]>4){n.tn(t,e,0,false);return;}for(A=t.e[e],c=false,D=0;D<A;D++)if(n.em(t,t.k[e][D])==1){c=true;break;}if(c){for(g=n.Gp(t,e),P=z(l.b7,jt,6,A,15,1),u=0;u<A;u++)P[u]=n.Xl(t,t.i[e][g[u]],e);for(o=0;o<A;o++)t.D[0][t.k[e][o]]==e&&n.em(t,t.k[e][o])==1&&(t.H[t.k[e][o]]=1);if(!((t.p&&e<t.f?n.ct(t.p,e):0)<=24&&n.tq(t,e,g,P))&&(h=n.oq(t,e,true),h!=-1)){for(t.D[0][h]!=e&&(t.D[1][h]=t.D[0][h],t.D[0][h]=e),b=-1,Q=0;Q<A;Q++)if(h==t.k[e][g[Q]]){b=Q;break;}for(d=F(k(l.d7,2),S1,7,0,[F(k(l.d7,1),H,6,15,[2,1,2,1]),F(k(l.d7,1),H,6,15,[1,2,2,1]),F(k(l.d7,1),H,6,15,[1,1,2,2]),F(k(l.d7,1),H,6,15,[2,1,1,2]),F(k(l.d7,1),H,6,15,[2,2,1,1]),F(k(l.d7,1),H,6,15,[1,2,1,2])]),i=1;i<A;i++)P[i]<P[0]&&(P[i]+=X1);if(A==3){switch(L=false,b){case 0:L=P[1]<P[2]&&P[2]-P[1]<n1||P[1]>P[2]&&P[1]-P[2]>n1;break;case 1:L=P[2]-P[0]>n1;break;case 2:L=P[1]-P[0]<n1;}r=(t.u[e]&3)==1^L?257:129;}else s=0,P[1]<=P[2]&&P[2]<=P[3]?s=0:P[1]<=P[3]&&P[3]<=P[2]?s=1:P[2]<=P[1]&&P[1]<=P[3]?s=2:P[2]<=P[3]&&P[3]<=P[1]?s=3:P[3]<=P[1]&&P[1]<=P[2]?s=4:P[3]<=P[2]&&P[2]<=P[1]&&(s=5),r=(t.u[e]&3)==1^d[s][b]==1?129:257;t.H[h]=r;}}}},n.wq=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O;if(!((t.F[e]&3)==0||(t.F[e]&3)==3||!n.Up(t,e))){for(S=-1,v=-1,M=-1,C=-1,r=0,s=0;s<2;s++)for(P=t.D[s][e],c=0;c<t.e[P];c++)u=t.k[P][c],u!=e&&n.em(t,u)==1&&(D=t.i[P][c],V=n.Ip(t,u,D),r<V&&(r=V,v=D,S=u,M=P,C=t.D[1-s][e]));if(v!=-1){for(h=0;h<2;h++)for(c=0;c<t.e[t.D[h][e]];c++)u=t.k[t.D[h][e]][c],u!=e&&n.em(t,u)==1&&(t.H[u]=1);for(t.D[1][S]!=v&&(t.D[0][S]=t.D[1][S],t.D[1][S]=v),o=St,b=0;b<t.j[M];b++)D=t.i[M][b],t.k[M][b]!=e&&o>D&&(o=D);for(d=z(l.d7,H,6,2,15,1),m=0,L=0;L<t.j[C];L++)t.k[C][L]!=e&&(d[m++]=t.i[C][L]);i=n.mo(t.J[M].a,t.J[M].b,t.J[C].a,t.J[C].b),A=0,m==2?(d[0]>d[1]&&(O=d[0],d[0]=d[1],d[1]=O),Q=n.no(i,n.Xl(t,C,d[0])),g=n.no(i,n.Xl(t,C,d[1])),A=Q-g):A=n.no(i,n.Xl(t,C,d[0])),A<0^(t.F[e]&3)==2^o==v?t.H[S]=257:t.H[S]=129;}}},n.xq=function(t){var e,A,P,r;for(n.aw(t,7),P=0;P<t.g;P++)(t.H[P]==257||t.H[P]==129)&&(t.H[P]=1);for(e=0;e<t.f;e++)n.vq(t,e);for(r=0;r<t.g;r++)n.wq(t,r);for(A=0;A<t.g;A++)t.H[A]==2&&(t.F[A]&3)==3&&(t.H[A]=386);},n.yq=function(t,e){var A,P,r,i,D,u,o,Q,L,s;if(u=z(l.d7,H,6,t.q,15,1),D=n.gp(t,u,false,e),D<=1)return null;for(o=z(l.d7,H,6,D,15,1),P=0;P<t.f;P++)++o[u[P]];for(L=0,s=o[0],Q=1;Q<D;Q++)s<o[Q]&&(s=o[Q],L=Q);for(A=0;A<t.q;A++)u[A]!=L&&(t.C[A]=-1);for(i=0;i<t.r;i++)(!e&&t.H[i]==16||u[t.D[0][i]]!=L)&&(t.H[i]=512);r=n.jl(t),t.T=0;try{n.yo(t,!0,!0);}catch(h){if(h=H1(h),!Jt(h,19))throw Ut(h);}return r;},n.zq=function(t,e){return (t.u[e]&be)!=0?true:t.C[e]<=1?false:n.Mm(t,e)||t.C[e]==13||t.C[e]>=171;},n.Aq=function(t){var e,A,P,r,i,D,u,o,Q,L;for(i=n.Vl(t,t.q,t.r,n.Pk),D=i*i/16,P=1;P<t.q;P++)for(r=0;r<P;r++)if(o=t.J[r].a-t.J[P].a,Q=t.J[r].b-t.J[P].b,L=t.J[r].c-t.J[P].c,o*o+Q*Q+L*L<D)throw Ut(new T.bC("The distance between two atoms is too close."));for(n.aw(t,1),e=0,A=0;A<t.f;A++){if(n.up(t,A)>(u=n.om(t,A),u+n.mm(t,A,u,t.C[A])))throw Ut(new T.bC("atom valence exceeded"));e+=t.s[A];}if(e!=0)throw Ut(new T.bC("unbalanced atom charge"));},n.Bq=function(t){var e,A;if(t.K)for(n.aw(t,7),e=0;e<t.f;e++)(t.u[e]&V0)!=0&&(M1(G(t.B[e],120),8)&&(t.B[e]=G(t.B[e],-9)),t.B[e]=G(t.B[e],-4294967297)),e<t.f&&n.nt(t.p,e)?t.B[e]=G(t.B[e],{l:4194297,m:W1,h:1048571}):kt(G(t.B[e],2),0)&&(t.B[e]=G(t.B[e],-5)),(kt(G(t.B[e],Be),0)||kt(G(t.B[e],Me),0)&&M1(G(t.B[e],Qe),0))&&M1(G(t.B[e],120),8)&&(t.B[e]=G(t.B[e],-9)),t.s[e]!=0&&(t.B[e]=G(t.B[e],-234881025)),n.up(t,e)==(A=n.om(t,e),A+n.mm(t,e,A,t.C[e]))&&(t.B[e]=G(t.B[e],-4063233),t.B[e]=G(t.B[e],-17042430230529),t.B[e]=G(t.B[e],-1921),t.B[e]=G(t.B[e],-114689)),kt(G(t.B[e],r0),0)&&t.w[e]!=0&&n._m(t,t.w[e]);},n.Cq=function(t){var e,A,P;if(t.K)for(n.aw(t,7),e=0;e<t.g;e++)P=Pn(t.G[e]&127),(e<t.g&&(n.tt(t.p,e)||t.H[e]==8))&(t.G[e]&8)!=0&&(t.G[e]&=-9,--P),P!=0&&(A=t.H[e]&127,A==1?t.G[e]|=1:A==2?t.G[e]|=2:A==4?t.G[e]|=4:A==32?t.G[e]|=32:A==64?t.G[e]|=64:A==16?t.G[e]|=16:A==8&&(t.G[e]|=8));},n.Dq=function(t,e,A){var P;return P=n._n(t,e,A),P&&A==386&&(n.aw(t,7),P=P&(t.F[e]&128)==0),P;},n.Eq=function(){n.co.call(this);},n.Fq=function(t,e){n.eo.call(this,t,e);},n.Gq=function(t){n.eo.call(this,t?t.M:256,t?t.N:256),t&&n.nl(t,this);},x(114,88,{114:1,88:1,4:1}),y.Zb=function(t){n.Do(this,t);},y.Yb=function(t){return n.Xp(this,t);},y.f=0,y.g=0,l.E7=X(114),n.Jq=function(t,e){return j.Math.pow(10,j.Math.log(2e3)*j.Math.LOG10E*t/(e-1)-1);},n.Kq=function(t,e){var A,P;for(A=e,P=0;e!=0;)t.d==0&&(t.f=(t.c[++t.e]&63)<<11,t.d=6),P|=(q1&t.f)>>16-A+e,t.f<<=1,--e,--t.d;return P;},n.Lq=function(t,e,A){t.d=6,t.e=A,t.c=e,t.f=(e[t.e]&63)<<11;},n.Mq=function(t,e){var A,P,r,i;return P=e/2|0,r=t>=P,r&&(t-=P),i=e/32,A=i*t/(P-t),r?-A:A;},n.Nq=function(t){var e,A,P,r;if(t.g.K)for(e=0;e<t.g.r;e++)P=n.gm(t.g,e),(P&127)!=0&&(A=-1,r=0,(P&16)!=0&&(A=16,++r),(P&64)!=0&&(A=64,++r),(P&32)!=0&&(A=32,++r),(P&4)!=0&&(A=4,++r),(P&2)!=0&&(A=2,++r),(P&8)!=0&&(A=8,++r),(P&1)!=0&&(A=1,++r),A!=-1&&(n.Ln(t.g,e,A),r==1&&n.Kn(t.g,e,127,false)));},n.Oq=function(t,e){var A;return e==null||w.Xzb(e).length==0?null:n.Qq(t,w.xzb((A=e,V1.Hmb(),A)));},n.Pq=function(t,e,A){var P,r;return e==null?null:n.Rq(t,w.xzb((r=e,V1.Hmb(),r)),A==null?null:w.xzb((P=A,P)),0);},n.Qq=function(t,e){var A;if(e==null||e.length==0)return null;for(A=2;A<e.length-2;A++)if(e[A]==32)return n.Rq(t,e,e,A+1);return n.Rq(t,e,null,0);},n.Rq=function(t,e,A,P){var r,i,D,u,o;return e==null?null:(n.Lq(t,e,0),r=n.Kq(t,4),u=n.Kq(t,4),r>8&&(r=u),i=n.Kq(t,r),D=n.Kq(t,u),o=new n.tw(i,D),n.Uq(t,o,e,A,P),o);},n.Sq=function(t,e,A){var P,r,i,D;if(A==null||w.Xzb(A).length==0){n.Tq(t,e,null,null);return;}P=T.ljb(A,Z1(32)),P>0&&P<w.Xzb(A).length-1?n.Tq(t,e,w.xzb((i=(w.Rzb(0,P,w.Xzb(A).length),w.Xzb(A).substr(0,P)),V1.Hmb(),i)),w.xzb((D=(w.Szb(P+1,w.Xzb(A).length+1),w.Xzb(A).substr(P+1)),D))):n.Tq(t,e,w.xzb((r=A,V1.Hmb(),r)),null);},n.Tq=function(t,e,A,P){if(A==null||A.length==0){n.il(e),e.d=null;return;}n.Uq(t,e,A,P,0);},n.Uq=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt,mt,y1,u1,Mt,Zt,m1,Et,f1,x1,s1,Q1,I1,G1,v0,K1,L1,D0,T0,j0,pe,ge,ne,Ae,R0,Re,K0,C0,Je,jn,zt,Wt,ze,Bn,mn,kn,i8,f8,D8,u8,o8,l8,Q8,L8,qP,VP,y8,U8,s8,h8,T8,b8,H0,c8,w8,g8,z8,d8,j8,B8,m8,k8,F8,oe,v8,rn,p8,_n,Pu,ru,iu,fu,Du,at,R8,uu,ou,lu,Ze,Qu,le,Wn,EP,RA,Lu,Pe,NP,yu,CA,Uu,HA,su,MA;if(n.il(e),e.d=null,!(A==null||0>=A.length)){if(t.g=e,NP=8,P!=null&&(r<0||r>=P.length)&&(P=null),n.Lq(t,A,0),u=n.Kq(t,4),O=n.Kq(t,4),u>8&&(NP=u,u=O),u==0){n.Nn(t.g,n.Kq(t,1)==1);return;}for(o=n.Kq(t,u),Q=n.Kq(t,O),Du=n.Kq(t,u),lu=n.Kq(t,u),uu=n.Kq(t,u),it=n.Kq(t,u),b=0;b<o;b++)n.Tk(t.g,6);for(T0=0;T0<Du;T0++)n.Bn(t.g,n.Kq(t,u),7);for(j0=0;j0<lu;j0++)n.Bn(t.g,n.Kq(t,u),8);for(zt=0;zt<uu;zt++)n.Bn(t.g,n.Kq(t,u),n.Kq(t,8));for(l8=0;l8<it;l8++)n.hn(t.g,n.Kq(t,u),n.Kq(t,4)-8);for(ct=1+Q-o,pt=n.Kq(t,4),V=0,n.yn(t.g,0,0),n.zn(t.g,0,0),n.An(t.g,0,0),mt=P!=null&&P[r]>=39,Pe=0,CA=0,HA=0,MA=0,Ht=false,xt=false,mt&&(P.length>2*o-2&&P[2*o-2]==39||P.length>3*o-3&&P[3*o-3]==39?(xt=true,Ht=P.length==3*o-3+9,oe=Ht?3*o-3:2*o-2,S=86*(P[oe+1]-40)+P[oe+2]-40,Pe=j.Math.pow(10,S/2e3-1),oe+=2,yu=86*(P[oe+1]-40)+P[oe+2]-40,CA=j.Math.pow(10,yu/1500-1),oe+=2,Uu=86*(P[oe+1]-40)+P[oe+2]-40,HA=j.Math.pow(10,Uu/1500-1),Ht&&(oe+=2,su=86*(P[oe+1]-40)+P[oe+2]-40,MA=j.Math.pow(10,su/1500-1))):Ht=P.length==3*o-3),t.$b()&&Ht&&(P=null,mt=false),H0=1;H0<o;H0++){if(Mt=n.Kq(t,pt),Mt==0){mt&&(n.yn(t.g,H0,n.Pl(t.g,0)+8*(P[H0*2-2]-83)),n.zn(t.g,H0,n.Ql(t.g,0)+8*(P[H0*2-1]-83)),Ht&&n.An(t.g,H0,n.Rl(t.g,0)+8*(P[2*o-3+H0]-83))),++ct;continue;}V+=Mt-1,mt&&(n.yn(t.g,H0,n.Pl(t.g,V)+P[H0*2-2]-83),n.zn(t.g,H0,n.Ql(t.g,V)+P[H0*2-1]-83),Ht&&n.An(t.g,H0,n.Rl(t.g,V)+(P[2*o-3+H0]-83))),n.Vk(t.g,V,H0,1);}for(j8=0;j8<ct;j8++)n.Vk(t.g,n.Kq(t,u),n.Kq(t,u),1);for(v8=z(l.zeb,ht,6,Q,16,1),Dt=0;Dt<Q;Dt++)switch(Lt=n.Kq(t,2),Lt){case 0:v8[Dt]=true;break;case 2:n.Ln(t.g,Dt,2);break;case 3:n.Ln(t.g,Dt,4);}for(D=n.Kq(t,u),B8=0;B8<D;B8++)if(h=n.Kq(t,u),NP==8)Ze=n.Kq(t,2),Ze==3?(n.nn(t.g,h,1,0),n.tn(t.g,h,1,false)):n.tn(t.g,h,Ze,false);else switch(Ze=n.Kq(t,3),Ze){case 4:n.tn(t.g,h,1,false),n.nn(t.g,h,1,n.Kq(t,3));break;case 5:n.tn(t.g,h,2,false),n.nn(t.g,h,1,n.Kq(t,3));break;case 6:n.tn(t.g,h,1,false),n.nn(t.g,h,2,n.Kq(t,3));break;case 7:n.tn(t.g,h,2,false),n.nn(t.g,h,2,n.Kq(t,3));break;default:n.tn(t.g,h,Ze,false);}for(NP==8&&n.Kq(t,1)==0&&(t.g.L=true),i=n.Kq(t,O),m8=0;m8<i;m8++)if(N=n.Kq(t,O),n.hm(t.g,N)==1)switch(Ze=n.Kq(t,3),Ze){case 4:n.In(t.g,N,1,false),n.Fn(t.g,N,1,n.Kq(t,3));break;case 5:n.In(t.g,N,2,false),n.Fn(t.g,N,1,n.Kq(t,3));break;case 6:n.In(t.g,N,1,false),n.Fn(t.g,N,2,n.Kq(t,3));break;case 7:n.In(t.g,N,2,false),n.Fn(t.g,N,2,n.Kq(t,3));break;default:n.In(t.g,N,Ze,false);}else n.In(t.g,N,n.Kq(t,2),false);for(n.Nn(t.g,n.Kq(t,1)==1),s=null,RA=null,R8=0;n.Kq(t,1)==1;)switch(Nt=R8+n.Kq(t,4),Nt){case 0:for(at=n.Kq(t,u),k8=0;k8<at;k8++)h=n.Kq(t,u),n.un(t.g,h,z1,true);break;case 1:for(at=n.Kq(t,u),F8=0;F8<at;F8++)h=n.Kq(t,u),iu=n.Kq(t,8),n.sn(t.g,h,iu);break;case 2:for(at=n.Kq(t,O),pe=0;pe<at;pe++)n.Kq(t,O);break;case 3:for(at=n.Kq(t,u),ge=0;ge<at;ge++)h=n.Kq(t,u),n.un(t.g,h,Y1,true);break;case 4:for(at=n.Kq(t,u),ne=0;ne<at;ne++)h=n.Kq(t,u),EP=b1(Ct(n.Kq(t,4)),3),n.un(t.g,h,EP,true);break;case 5:for(at=n.Kq(t,u),Ae=0;Ae<at;Ae++)h=n.Kq(t,u),L=b1(Ct(n.Kq(t,2)),1),n.un(t.g,h,L,true);break;case 6:for(at=n.Kq(t,u),R0=0;R0<at;R0++)h=n.Kq(t,u),n.un(t.g,h,1,true);break;case 7:for(at=n.Kq(t,u),Re=0;Re<at;Re++)h=n.Kq(t,u),K1=b1(Ct(n.Kq(t,4)),7),n.un(t.g,h,K1,true);break;case 8:for(at=n.Kq(t,u),K0=0;K0<at;K0++){for(h=n.Kq(t,u),C=n.Kq(t,4),d=z(l.d7,H,6,C,15,1),rn=0;rn<C;rn++)m=n.Kq(t,8),d[rn]=m;n.on(t.g,h,d);}break;case 9:for(at=n.Kq(t,O),C0=0;C0<at;C0++)N=n.Kq(t,O),EP=n.Kq(t,2)<<7,n.Kn(t.g,N,EP,true);break;case 10:for(at=n.Kq(t,O),Je=0;Je<at;Je++)N=n.Kq(t,O),rt=n.Kq(t,5),n.Kn(t.g,N,rt,true);break;case 11:for(at=n.Kq(t,u),jn=0;jn<at;jn++)h=n.Kq(t,u),n.un(t.g,h,Q0,true);break;case 12:for(at=n.Kq(t,O),Wt=0;Wt<at;Wt++)N=n.Kq(t,O),yt=n.Kq(t,8)<<9,n.Kn(t.g,N,yt,true);break;case 13:for(at=n.Kq(t,u),ze=0;ze<at;ze++)h=n.Kq(t,u),Qu=b1(Ct(n.Kq(t,3)),14),n.un(t.g,h,Qu,true);break;case 14:for(at=n.Kq(t,u),Bn=0;Bn<at;Bn++)h=n.Kq(t,u),fu=b1(Ct(n.Kq(t,5)),17),n.un(t.g,h,fu,true);break;case 15:case 31:R8+=16;break;case 16:for(at=n.Kq(t,u),mn=0;mn<at;mn++)h=n.Kq(t,u),Wn=b1(Ct(n.Kq(t,3)),22),n.un(t.g,h,Wn,true);break;case 17:for(at=n.Kq(t,u),kn=0;kn<at;kn++)h=n.Kq(t,u),n.fn(t.g,h,n.Kq(t,4));break;case 18:for(at=n.Kq(t,u),ru=n.Kq(t,4),i8=0;i8<at;i8++){for(h=n.Kq(t,u),Ot=n.Kq(t,ru),p8=z(l._6,J,6,Ot,15,1),rn=0;rn<Ot;rn++)p8[rn]=n.Kq(t,7)<<24>>24;n.ln(t.g,h,T.cjb((_n=p8,V1.Hmb(),_n)));}break;case 19:for(at=n.Kq(t,u),f8=0;f8<at;f8++)h=n.Kq(t,u),bt=b1(Ct(n.Kq(t,3)),25),n.un(t.g,h,bt,true);break;case 20:for(at=n.Kq(t,O),D8=0;D8<at;D8++)N=n.Kq(t,O),Wn=n.Kq(t,3)<<17,n.Kn(t.g,N,Wn,true);break;case 21:for(at=n.Kq(t,u),u8=0;u8<at;u8++)h=n.Kq(t,u),n.vn(t.g,h,n.Kq(t,2)<<4);break;case 22:for(at=n.Kq(t,u),o8=0;o8<at;o8++)h=n.Kq(t,u),n.un(t.g,h,Ie,true);break;case 23:for(at=n.Kq(t,O),Q8=0;Q8<at;Q8++)N=n.Kq(t,O),n.Kn(t.g,N,se,true);break;case 24:for(at=n.Kq(t,O),L8=0;L8<at;L8++)N=n.Kq(t,O),L=n.Kq(t,2)<<21,n.Kn(t.g,N,L,true);break;case 25:for(qP=0;qP<o;qP++)n.Kq(t,1)==1&&n.wn(t.g,qP,true);break;case 26:for(at=n.Kq(t,O),s=z(l.d7,H,6,at,15,1),VP=0;VP<at;VP++)s[VP]=n.Kq(t,O);break;case 27:for(at=n.Kq(t,u),y8=0;y8<at;y8++)h=n.Kq(t,u),n.un(t.g,h,r0,true);break;case 28:for(at=n.Kq(t,O),U8=0;U8<at;U8++)n.Ln(t.g,n.Kq(t,O),16);break;case 29:for(at=n.Kq(t,u),s8=0;s8<at;s8++)h=n.Kq(t,u),v0=b1(Ct(n.Kq(t,2)),30),n.un(t.g,h,v0,true);break;case 30:for(at=n.Kq(t,u),h8=0;h8<at;h8++)h=n.Kq(t,u),Wn=b1(Ct(n.Kq(t,7)),32),n.un(t.g,h,Wn,true);break;case 32:for(at=n.Kq(t,u),T8=0;T8<at;T8++)h=n.Kq(t,u),Lu=b1(Ct(n.Kq(t,2)),44),n.un(t.g,h,Lu,true);break;case 33:for(at=n.Kq(t,u),b8=0;b8<at;b8++)h=n.Kq(t,u),x1=b1(Ct(n.Kq(t,5)),39),n.un(t.g,h,x1,true);break;case 34:for(at=n.Kq(t,u),c8=0;c8<at;c8++)h=n.Kq(t,u),n.un(t.g,h,AA,true);break;case 35:for(at=n.Kq(t,O),w8=0;w8<at;w8++)N=n.Kq(t,O),n.Kn(t.g,N,yn,true);break;case 36:for(at=n.Kq(t,O),g8=0;g8<at;g8++)N=n.Kq(t,O),ot=n.Kq(t,2)<<5,n.Kn(t.g,N,ot,true);break;case 37:for(at=n.Kq(t,O),z8=0;z8<at;z8++)N=n.Kq(t,O),ot=n.Kq(t,1)==0?32:64,n.Ln(t.g,N,ot);break;case 38:for(at=n.Kq(t,u),wt=n.Kq(t,3),RA=z(l.d7,H,6,o,15,1),d8=0;d8<at;d8++)h=n.Kq(t,u),$t=n.Kq(t,wt),RA[h]=$t;break;case 39:for(at=n.Kq(t,u),D0=0;D0<at;D0++)h=n.Kq(t,u),ou=b1(Ct(n.Kq(t,4)),47),n.un(t.g,h,ou,true);}if(n.Fe(new n.Ue(t.g),v8,false),s!=null)for(I=s,ut=0,Qt=I.length;ut<Qt;++ut)N=I[ut],n.Ln(t.g,N,n.hm(t.g,N)==2?4:2);if(P==null&&!t.i&&A.length>t.e+1&&(A[t.e+1]==32||A[t.e+1]==9)&&(P=A,r=t.e+2),P!=null)try{if(P[r]==33||P[r]==35){for(n.Lq(t,P,r+1),Ht=n.Kq(t,1)==1,xt=n.Kq(t,1)==1,le=2*n.Kq(t,4),K=1<<le,I1=0,N=0,c=1;c<o;c++)N<Q&&n.Yl(t.g,1,N)==c?(I1=n.Yl(t.g,0,N++),Q1=1):(I1=0,Q1=8),y1=Q1*(n.Kq(t,le)+1-(K>>1)),u1=Q1*(n.Kq(t,le)+1-(K>>1)),n.yn(t.g,c,n.Pl(t.g,I1)+y1),n.zn(t.g,c,n.Ql(t.g,I1)+u1),Ht&&n.An(t.g,c,n.Rl(t.g,I1)+Q1*(n.Kq(t,le)+1-(K>>1)));if(P[r]==35){for(L1=0,G1=z(l.d7,H,6,o,15,1),g=0;g<o;g++)L1+=G1[g]=n.np(t.g,g);for(h=0;h<o;h++)for(D0=0;D0<G1[h];D0++)K1=n.Tk(t.g,1),n.Vk(t.g,h,K1,1),n.yn(t.g,K1,n.Pl(t.g,h)+(n.Kq(t,le)+1-(K>>1))),n.zn(t.g,K1,n.Ql(t.g,h)+(n.Kq(t,le)+1-(K>>1))),Ht&&n.An(t.g,K1,n.Rl(t.g,h)+(n.Kq(t,le)+1-(K>>1))),RA!=null&&(RA[h]&1<<D0)!=0&&n.wn(t.g,K1,!0);o+=L1,Q+=L1;}if(M=Ht?1.5:(n.Qk(),n.Qk(),n.Pk),v=n.Vl(t.g,o,Q,M),xt)for(Pe=n.Jq(n.Kq(t,le),K),CA=Pe*n.Mq(n.Kq(t,le),K),HA=Pe*n.Mq(n.Kq(t,le),K),Ht&&(MA=Pe*n.Mq(n.Kq(t,le),K)),Q1=Pe/v,h=0;h<o;h++)n.yn(t.g,h,CA+Q1*n.Pl(t.g,h)),n.zn(t.g,h,HA+Q1*n.Ql(t.g,h)),Ht&&n.An(t.g,h,MA+Q1*n.Rl(t.g,h));else for(Pe=1.5,Q1=Pe/v,h=0;h<o;h++)n.yn(t.g,h,Q1*n.Pl(t.g,h)),n.zn(t.g,h,Q1*n.Ql(t.g,h)),Ht&&n.An(t.g,h,Q1*n.Rl(t.g,h));}else if(Ht&&!xt&&Pe==0&&(Pe=1.5),Pe!=0&&t.g.r!=0){for(v=0,N=0;N<t.g.r;N++)Zt=n.Pl(t.g,n.Yl(t.g,0,N))-n.Pl(t.g,n.Yl(t.g,1,N)),m1=n.Ql(t.g,n.Yl(t.g,0,N))-n.Ql(t.g,n.Yl(t.g,1,N)),Et=Ht?n.Rl(t.g,n.Yl(t.g,0,N))-n.Rl(t.g,n.Yl(t.g,1,N)):0,v+=j.Math.sqrt(Zt*Zt+m1*m1+Et*Et);for(v/=t.g.r,s1=Pe/v,h=0;h<t.g.q;h++)n.yn(t.g,h,n.Pl(t.g,h)*s1+CA),n.zn(t.g,h,n.Ql(t.g,h)*s1+HA),Ht&&n.An(t.g,h,n.Rl(t.g,h)*s1+MA);}}catch(de){if(de=H1(de),Jt(de,19))f1=de,T.LB(f1,(T.Ujb(),T.Tjb),""),""+f1+T.cjb((Pu=A,V1.Hmb(),Pu))+T.cjb((_n=P,_n)),P=null,Ht=false;else throw Ut(de);}if(vt=P!=null&&!Ht,n.Nq(t),vt||t.$b())for(n.aw(t.g,7),N=0;N<t.g.g;N++)n.em(t.g,N)==2&&!n.iq(t.g,N)&&n.fm(t.g,N)==0&&n.Jn(t.g,N);if(n.uq(t.g,0),!vt&&t.$b())try{t._b(t.g),vt=!0;}catch(de){if(de=H1(de),Jt(de,19))f1=de,T.LB(f1,(T.Ujb(),T.Tjb),""),""+f1+T.cjb((_n=A,V1.Hmb(),_n));else throw Ut(de);}vt?(n.xq(t.g),n.nw(t.g)):Ht||n.uq(t.g,0);}},n.Vq=function(t,e){var A,P,r,i,D,u;if(!(e.length<=0||e[0]<64))for(n.Lq(t,e,0),u=n.Kq(t,4),r=n.Kq(t,1)==1,i=n.Kq(t,1)==1,A=0;A<t.g.f;A++)D=n.Kq(t,u),P=r,r&&i&&(P=n.Kq(t,1)==1),n.qn(t.g,A,D,P);},n.Wq=function(){},x(312,1,{},n.Wq),y.$b=function(){return  false;},y._b=function(t){throw Ut(new T.bC("Unexpected request to invent coordinates. Check source code logic!"));},y.d=0,y.e=0,y.f=0,y.i=false,l.F7=X(312),n.Zq=function(t){},n.$q=function(t){n.Zq(this),this.b=t;},x(50,312,{},n.$q),y.$b=function(){return this.b;},y._b=function(t){var e;e=new Z.BA(this.a),e.g=new U.zvb(78187493520),Z.iA(e,t);},y.a=2,y.b=false,l.G7=X(50),n.br=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m;r=e.a,b=e.g,Q=kt(G(n.Nl(t.k,r),1),0),i=n.Il(t.k,r),s=i!=null?n.ir(t,r,i,P):Q?"*":n.Hl(t.k,r),!Q&&i==null&&n.Sp(t.k,r)&&(t.j&4)==0&&(n.Po(t.k,r)!=0||n.yl(t.k,r)==-1&&n.Ol(t.k,r)==0)&&(s=w.Xzb(s).toLowerCase()),e.f&&(A.a+="("),b!=-1&&n.fr(t,n.Wo(t.k,e.a,e.g),e.g,A),D=n.Al(t.k,r),D==0&&(t.j&1)!=0&&(u=G(n.Nl(t.k,r),je),et(u,ln)==0?D=-1:et(u,on)==0&&(D=1)),L=n.Ll(t.k,r),h=(t.j&2)!=0?n.Kl(t.k,r):0,c=(t.j&1)!=0?n.or(t,r,P):null,m=!Q&&!n.vr(n.Sl(t.k,r))||i!=null||(n.Ml(t.k,r)==1||n.Ml(t.k,r)==2)&&(d=n.Gl(t.k,r)-1,!(d!=-1&&t.i[d][n.Fl(t.k,r)]<=1))&&(n.Sl(t.k,r)!=7||n.Al(t.k,r)>0)||n.Sp(t.k,r)&&n.Po(t.k,r)==0&&(t.j&4)==0||D!=0||L!=0||h!=0||n.yl(t.k,r)!=-1||n.Ol(t.k,r)!=0||c!=null,m&&(A.a+="["),L!=0&&(A.a+=L),A.a+=""+s,(n.Ml(t.k,r)==1||n.Ml(t.k,r)==2)&&(g=n.Gl(t.k,r)-1,!(g!=-1&&t.i[g][n.Fl(t.k,r)]<=1))&&(n.Sl(t.k,r)!=7||n.Al(t.k,r)>0)&&T.Njb(A,n.nr(t,r,b)),(t.j&1)==0&&m&&(o=n.Bp(t.k,r),o==1?A.a+="H":o>1&&(A.a+="H"+o)),D!=0&&(A.a+=String.fromCharCode(D>0?43:45),j.Math.abs(D)>1&&T.Kjb(A,j.Math.abs(D))),c!=null&&(A.a+=""+c),h!=0&&(A.a+=":",A.a+=h),m&&(A.a+="]"),n.gr(t,e,A),e.e&&(A.a+=")");},n.cr=function(t,e,A){var P,r,i,D,u;for(i=true,u=U.wj(t.g,e).a,D=n.rr(t,u);D!=-1;)P=n.Yo(t.k,u,D),r=n.$o(t.k,u,D),D=n.rr(t,P),n.er(t,new n.Zu(P,r,u,A&&i,A&&D==-1),++e),u=P,i=false;},n.dr=function(t,e,A){var P,r,i,D;for(r=e,i=0,D=r.length;i<D;++i)P=r[i],t.f[P&1073741823]=(P&E0)!=0^A?2:1;},n.er=function(t,e,A){U.qj(t.g,A,e),t.b[e.a]=true,e.g!=-1&&(t.c[n.Wo(t.k,e.a,e.g)]=true);},n.fr=function(t,e,A,P){var r,i,D,u,o;o=w.Xzb(P.a).length,t.f[e]!=0&&T.Jjb(P,t.f[e]==1?47:92),t.j==1&&(r=n.gm(t.k,e)&127,r!=0&&((r&1)!=0&&t.f[e]==0&&(P.a+="-"),(r&2)!=0&&(w.Xzb(P.a).length!=o&&(P.a+=","),P.a+="="),(r&4)!=0&&(w.Xzb(P.a).length!=o&&(P.a+=","),P.a+="#"),(r&32)!=0&&(w.Xzb(P.a).length!=o&&(P.a+=","),P.a+="$"),(r&64)!=0&&(w.Xzb(P.a).length!=o&&(P.a+=","),P.a+="$"),(r&8)!=0&&(w.Xzb(P.a).length!=o&&(P.a+=","),P.a+=":"),(r&16)!=0&&(w.Xzb(P.a).length!=o&&(P.a+=","),T.Njb(P,n.Km(t.k,A)?"<-":"->")))),o==w.Xzb(P.a).length&&(!n.Tp(t.k,e)||(t.j&4)!=0)&&(i=n.hm(t.k,e)&127,i==1?n.Sp(t.k,n.Yl(t.k,0,e))&&n.Sp(t.k,n.Yl(t.k,1,e))&&(t.j&4)==0&&t.f[e]==0&&(P.a+="-"):i==2?P.a+="=":i==4?P.a+="#":i==32||i==64?P.a+="$":i==8?P.a+=":":i==16&&T.Njb(P,n.Km(t.k,A)?"<-":"->")),t.j==1&&(D=o==w.Xzb(P.a).length?"":";",u=n.gm(t.k,e)&384,u==256?P.a+=D+"@":u==128&&(P.a+=D+"!@"));},n.gr=function(t,e,A){var P,r,i;if(e.c!=null)for(r=0;r<e.c.length;r++)for(i=0;i<n.Zo(t.k,e.a);i++)e.c[r]==n.Yo(t.k,e.a,i)&&(P=n.$o(t.k,e.a,i),e.d[r]||n.fr(t,P,e.a,A),t.e[P]>9&&(A.a+="%"),T.Kjb(A,t.e[P]));},n.hr=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct;for(bt=new U.Lj(),m=new U.wqb(t.g);m.a<m.c.a.length;)if(d=U.vqb(m),d.g!=-1&&(v=n.Wo(t.k,d.a,d.g),!n.Up(t.k,v)&&!n.iq(t.k,v)&&(n.fm(t.k,v)==1||n.fm(t.k,v)==2))){if(rt=U.wj(t.g,t.q[d.g]),A=z(l.d7,H,6,n.Zo(t.k,d.a)+n.Zo(t.k,rt.a)-2,15,1),K=0,yt=false,rt.g!=-1)A[K++]=rt.b;else {for(M=-1,it=-1,S=St,I=0;I<n.Zo(t.k,rt.a);I++)s=n.Yo(t.k,rt.a,I),s!=d.a&&(M==-1?(M=I,S=t.q[s]):S<t.q[s]?it=I:(it=M,M=I));it==-1?(Dt=n.Yo(t.k,rt.a,M),Lt=n.$o(t.k,rt.a,M),A[K++]=Lt|(n.tr(t,rt.a,Dt)?E0:0)):(h=n.Yo(t.k,rt.a,M),c=n.$o(t.k,rt.a,M),b=n.Yo(t.k,rt.a,it),g=n.$o(t.k,rt.a,it),A[K++]=c|(n.tr(t,rt.a,h)?E0:0),A[K++]=g|(n.tr(t,rt.a,b)?0:E0));}if(n.Zo(t.k,rt.a)==3&&rt.g!=-1){for(I=0;I<n.Zo(t.k,rt.a);I++)if(s=n.Yo(t.k,rt.a,I),s!=rt.g&&s!=d.a){P=n.$o(t.k,rt.a,I),A[K++]=P|(n.tr(t,rt.a,s)?E0:0),s<rt.g&&(yt=!yt);break;}}for(n.fm(t.k,v)==2&&(yt=!yt),N=0;N<n.Zo(t.k,d.a);N++)if(o=n.Yo(t.k,d.a,N),o!=d.g){if(O=yt,n.Zo(t.k,d.a)==3){for(Qt=0;Qt<n.Zo(t.k,d.a);Qt++)if(s=n.Yo(t.k,d.a,Qt),s!=d.g&&s!=o){s<o&&(O=!O);break;}}n.Em(t.k,v)&&(V=n.sg(t.d,v),t.n[V]||(t.n[V]=true,t.o[V]=O),t.o[V]&&(O=!O)),Q=n.Wo(t.k,d.a,o),A[K++]=Q|(O^n.tr(t,d.a,o)?0:E0);}w.nzb(bt.a,A);}for(t.f=z(l.d7,H,6,t.k.g,15,1),bt.a.length!=0&&n.dr(t,U.Dj(bt,0),false);bt.a.length!=0;){for(ct=bt.a.length,N=bt.a.length-1;N>=0;N--){for(A=(w.Kzb(N,bt.a.length),bt.a[N]),ot=0,ut=false,L=false,i=A,D=0,u=i.length;D<u;++D)r=i[D],e=r&1073741823,t.f[e]!=0&&(C=(r&E0)!=0^t.f[e]==2,ot==0?ut=C:ut!=C&&(L=true),++ot);ot!=0&&(A=U.Dj(bt,N),L||n.dr(t,A,ut));}ct==bt.a.length&&n.dr(t,U.Dj(bt,0),false);}},n.ir=function(t,e,A,P){var r,i,D,u,o,Q,L;for(L=w.Xzb(P.a).length,0<L?P.a=T.vjb(P.a,0,0):0>L&&(P.a+=T.Bjb(z(l.a7,n0,6,-L,15,1))),o=n.Sp(t.k,e)&&(t.j&4)==0,i=A,D=0,u=i.length;D<u;++D)r=i[D],w.Xzb(P.a).length!=0&&(P.a+=","),Q=(n.Qk(),n.Kk)[r],T.Njb(P,o?w.Xzb(Q).toLowerCase():Q);return P.a;},n.jr=function(t){var e,A,P,r,i,D,u,o;if(!t.k||t.k.q==0)return "";for(n.aw(t.k,15),t.d=new n.Ng(t.k,129),r=t.d.T,t.o=z(l.zeb,ht,6,r+1,16,1),t.n=z(l.zeb,ht,6,r+1,16,1),t.i=h1(l.d7,[S1,H],[7,6],15,[2,32],2),e=0;e<t.k.f;e++)o=n.Gl(t.k,e)-1,o!=-1&&++t.i[o][n.Fl(t.k,e)];for(n.mr(t),n.kr(t),n.hr(t),P=new T.Pjb(),A=new T.Pjb(),i=true,u=new U.wqb(t.g);u.a<u.c.a.length;)D=U.vqb(u),D.g==-1&&(i?i=false:P.a+="."),n.br(t,D,P,A);return P.a;},n.kr=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(s=new U.wqb(t.g);s.a<s.c.a.length;){for(Q=U.vqb(s),A=0,i=0;i<n.Zo(t.k,Q.a);i++)t.c[n.$o(t.k,Q.a,i)]||++A;if(A!=0){for(Q.c=z(l.d7,H,6,A,15,1),A=0,D=0;D<n.Zo(t.k,Q.a);D++)t.c[n.$o(t.k,Q.a,D)]||(o=n.Yo(t.k,Q.a,D),Q.c[A++]=t.q[o]<<16|o);for(w.qzb(Q.c,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),r=0;r<Q.c.length;r++)Q.c[r]=r1&Q.c[r];}}for(P=z(l.zeb,ht,6,t.k.g,16,1),t.e=z(l.d7,H,6,t.k.g,15,1),L=new U.wqb(t.g);L.a<L.c.a.length;)if(Q=U.vqb(L),Q.c!=null){for(Q.d=z(l.zeb,ht,6,Q.c.length,16,1),r=0;r<Q.c.length;r++)for(u=0;u<n.Zo(t.k,Q.a);u++)if(Q.c[r]==n.Yo(t.k,Q.a,u))if(e=n.$o(t.k,Q.a,u),t.c[e])P[t.e[e]]=false;else {for(t.c[e]=true,Q.d[r]=true,t.e[e]=1;P[t.e[e]];)++t.e[e];P[t.e[e]]=true;}}},n.lr=function(t){var e,A,P,r;for(P=-1,r=St,e=0;e<t.k.f;e++)t.b[e]||(A=t.a[e],n.Im(t.k,e)&&(A+=E0),n.Zo(t.k,e)==0?A+=1056964608:A+=n.Zo(t.k,e)<<24,r>A&&(r=A,P=e));return P;},n.mr=function(t){var e,A,P,r,i;for(t.a=t.d.f,t.b=z(l.zeb,ht,6,t.k.f,16,1),t.c=z(l.zeb,ht,6,t.k.g,16,1),t.g=new U.Lj(),e=n.lr(t);e!=-1;){if(A=t.g.a.length,n.er(t,new n.Zu(e,-1,-1,false,false),A),n.Zo(t.k,e)!=0)for(n.cr(t,A,false);A<t.g.a.length-1;){for(;n.sr(t,U.wj(t.g,A).a);)n.cr(t,A,true);++A;}e=n.lr(t);}for(t.q=z(l.d7,H,6,t.k.f,15,1),P=0,i=new U.wqb(t.g);i.a<i.c.a.length;)r=U.vqb(i),t.q[r.a]=P++;},n.nr=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;if(o=false,n.Po(t.k,e)!=0&&n.Zo(t.k,e)==2&&n._o(t.k,e,0)==2&&n._o(t.k,e,1)==2){for(D=0;D<n.Zo(t.k,e);D++)if(P=n.Yo(t.k,e,D),P!=A&&n.Zo(t.k,P)==2){o=true;break;}for(i=0;i<n.Zo(t.k,e);i++){for(P=n.Yo(t.k,e,i),c=0,b=z(l.d7,H,6,3,15,1),L=0;L<n.Zo(t.k,P);L++)b[c]=n.Yo(t.k,P,L),b[c]!=e&&++c;c==2&&t.q[b[0]]<t.q[b[1]]^b[0]<b[1]&&(o=!o);}}else {for(s=z(l.d7,H,6,4,15,1),h=z(l.d7,H,6,4,15,1),u=0,A!=-1&&(s[u]=A,h[u++]=8*t.q[A]),(n.np(t.k,e)!=0||n.Zo(t.k,e)==3)&&(s[u]=St,h[u++]=8*t.q[e]),i=0;i<n.Zo(t.k,e);i++)P=n.Yo(t.k,e,i),P!=A&&(s[u]=P,h[u++]=n.qr(t,e,i));o=n.ur(s,h);}return Q=n.Ml(t.k,e)==1^o,n.xm(t.k,e)&&(r=n.tg(t.d,e),t.n[r]||(t.n[r]=true,t.o[r]=Q),t.o[r]&&(Q=!Q)),Q?"@@":"@";},n.or=function(t,e,A){var P,r,i,D,u,o,Q,L;switch(u=w.Xzb(A.a).length,0<u?A.a=T.vjb(A.a,0,0):0>u&&(A.a+=T.Bjb(z(l.a7,n0,6,-u,15,1))),o=n.Nl(t.k,e),r=Gt(t0(G(o,je),3)),r){case 20971520:A.a+="+0";break;case 25165824:n.Al(t.k,e)==0&&(A.a+="-");break;case 12582912:n.Al(t.k,e)==0&&(A.a+="+");}return P=G(o,ZA),et(P,2)==0?A.a+=";a":et(P,4)==0&&(A.a+=";A"),i=G(o,Z0),et(i,0)!=0&&(et(i,1792)==0?A.a+=";H0":et(i,1664)==0?A.a+=";H1":et(i,1408)==0?A.a+=";H2":et(i,896)==0?A.a+=";H3":et(i,128)==0?A.a+=";!H0":et(i,384)==0?A.a+=";!H0;!H1":et(i,1536)==0?A.a+=";!H2;!H3":et(i,i0)==0&&(A.a+=";!H3")),L=G(o,120),et(L,8)==0?A.a+=";!R0":et(L,16)==0?A.a+=";!R1":et(L,32)==0?A.a+=";!R2":et(L,64)==0?A.a+=";!R3":et(L,112)==0?A.a+=";R0":et(L,104)==0?A.a+=";R1":et(L,88)==0?A.a+=";R2":et(L,56)==0&&(A.a+=";R3"),Q=G(o,Me),et(Q,Qe)==0?A.a+=";!r"+y0(Q):et(Q,K8)==0?A.a+=";r"+y0(Q):et(Q,0)!=0&&(kt(G(Q,un),0)?(M1(G(Q,Qe),0)&&(A.a+=";!r0"+y0(Q)),M1(G(Q,Se),0)&&(A.a+=";!r3"+y0(Q)),M1(G(Q,qe),0)&&(A.a+=";!r4"+y0(Q)),M1(G(Q,Ve),0)&&(A.a+=";!r5"+y0(Q)),M1(G(Q,Ee),0)&&(A.a+=";!r6"+y0(Q)),M1(G(Q,Ne),0)&&(A.a+=";!r7"+y0(Q))):(A.a+=";",kt(G(Q,Qe),0)&&(A.a+="r0,"+y0(Q)),kt(G(Q,Se),0)&&(A.a+="r3,"+y0(Q)),kt(G(Q,qe),0)&&(A.a+="r4,"+y0(Q)),kt(G(Q,Ve),0)&&(A.a+="r5,"+y0(Q)),kt(G(Q,Ee),0)&&(A.a+="r6,"+y0(Q)),kt(G(Q,Ne),0)&&(A.a+="r7,"+y0(Q)),T.Agb(A,w.Xzb(A.a).length-1))),et(Q,0)==0&&(Q=t0(G(o,Be),22),et(Q,0)!=0&&(A.a+=";r"+y0(Q))),D=G(o,ae),et(D,rA)==0&&(A.a+=";D1"),et(D,pn)==0&&(A.a+=";D2"),et(D,Rn)==0&&(A.a+=";D3"),et(D,iA)==0&&(A.a+=";!D3;!D4"),et(D,F0)==0&&(A.a+=";!D4"),et(D,tn)==0&&(A.a+=";!D0;!D1"),et(D,ie)==0&&(A.a+=";!D0;!D1;!D2"),et(D,Qn)==0&&(A.a+=";!D0;!D1;!D2;!D3"),kt(G(o,z1),0)&&T.Njb(A,";D"+n.Zo(t.k,e)),kt(G(o,Y1),0)&&T.Njb(A,";!D"+n.Zo(t.k,e)),w.Xzb(A.a).length==0?null:A.a;},n.pr=function(t){return t.p==null&&(t.p=n.jr(t)),t.p;},n.qr=function(t,e,A){var P,r,i,D,u;if(P=n.$o(t.k,e,A),D=n.Yo(t.k,e,A),t.e[P]!=0){for(u=8*t.q[e]+1,r=U.wj(t.g,t.q[e]).c,i=0;i<r.length&&D!=r[i];i++)++u;return u;}return 8*t.q[D];},n.rr=function(t,e){var A,P,r,i,D;for(A=-1,P=-1,i=0;i<n.Zo(t.k,e);i++)r=n.Yo(t.k,e,i),D=(n._o(t.k,e,i)<<24)+t.a[r],!t.b[r]&&(A==-1||P<D)&&(A=i,P=D);return A;},n.sr=function(t,e){var A;for(A=0;A<n.Zo(t.k,e);A++)if(!t.b[n.Yo(t.k,e,A)])return  true;return  false;},n.tr=function(t,e,A){var P,r;return P=U.wj(t.g,t.q[e]),P.g==A?false:(r=U.wj(t.g,t.q[A]),r.g==e?true:n.Yu(r,e));},n.ur=function(t,e){var A,P,r;for(P=false,A=1;A<4;A++)for(r=0;r<A;r++)t[r]>t[A]&&(P=!P),e[r]>e[A]&&(P=!P);return P;},n.vr=function(t){return t>=5&&t<=9||t>=15&&t<=17||t==35||t==53;},n.wr=function(t){n.xr.call(this,t,0);},n.xr=function(t,e){this.k=t,this.j=e;},n.yr=function(t){var e,A,P,r;for(r=new T.Pjb(),A=0;A<t.g.a.length;A++)A!=0&&(r.a+="."),T.Njb(r,n.pr(new n.xr(U.wj(t.g,A),2)));for(r.a+=">",P=0;P<t.a.a.length;P++)P!=0&&(r.a+="."),T.Njb(r,n.pr(new n.wr(U.wj(t.a,P))));for(r.a+=">",e=0;e<t.f.a.length;e++)e!=0&&(r.a+="."),T.Njb(r,n.pr(new n.xr(U.wj(t.f,e),2)));return r.a;},x(111,1,{},n.wr,n.xr),y.j=0,l.H7=X(111),n.zr=function(t,e){this.b=t,this.a=e;},x(3,1,{3:1},n.zr),y.a=0,y.b=0,l.I7=X(3),n.Br=function(){n.Br=o1,n.Ar=F(k(l.I7,2),c0,17,0,[null,F(k(l.I7,1),qt,3,0,[new n.zr(0,1.007825032),new n.zr(1,2.014101778),new n.zr(2,3.016049268),new n.zr(3,4.027834627),new n.zr(4,5.039542911),new n.zr(5,6.044942608)]),F(k(l.I7,1),qt,3,0,[new n.zr(1,3.01602931),new n.zr(2,4.00260325),new n.zr(3,5.012223628),new n.zr(4,6.018888072),new n.zr(5,7.028030527),new n.zr(6,8.033921838),new n.zr(7,9.043820323),new n.zr(8,10.052399713)]),F(k(l.I7,1),qt,3,0,[new n.zr(1,4.027182329),new n.zr(2,5.012537796),new n.zr(3,6.015122281),new n.zr(4,7.016004049),new n.zr(5,8.02248667),new n.zr(6,9.026789122),new n.zr(7,10.035480884),new n.zr(8,11.043796166),new n.zr(9,12.05378)]),F(k(l.I7,1),qt,3,0,[new n.zr(1,5.04079),new n.zr(2,6.019725804),new n.zr(3,7.016929246),new n.zr(4,8.005305094),new n.zr(5,9.012182135),new n.zr(6,10.01353372),new n.zr(7,11.021657653),new n.zr(8,12.026920631),new n.zr(9,13.036133834),new n.zr(10,14.042815522)]),F(k(l.I7,1),qt,3,0,[new n.zr(2,7.029917389),new n.zr(3,8.024606713),new n.zr(4,9.013328806),new n.zr(5,10.012937027),new n.zr(6,11.009305466),new n.zr(7,12.014352109),new n.zr(8,13.017780267),new n.zr(9,14.025404064),new n.zr(10,15.031097291),new n.zr(11,16.039808836),new n.zr(12,17.046931399),new n.zr(13,18.05617),new n.zr(14,19.06373)]),F(k(l.I7,1),qt,3,0,[new n.zr(2,8.037675026),new n.zr(3,9.031040087),new n.zr(4,10.01685311),new n.zr(5,11.011433818),new n.zr(6,12),new n.zr(7,13.003354838),new n.zr(8,14.003241988),new n.zr(9,15.010599258),new n.zr(10,16.014701243),new n.zr(11,17.022583712),new n.zr(12,18.026757058),new n.zr(13,19.035248094),new n.zr(14,20.040322395),new n.zr(15,21.04934),new n.zr(16,22.05645)]),F(k(l.I7,1),qt,3,0,[new n.zr(3,10.042618),new n.zr(4,11.026796226),new n.zr(5,12.018613202),new n.zr(6,13.005738584),new n.zr(7,14.003074005),new n.zr(8,15.000108898),new n.zr(9,16.006101417),new n.zr(10,17.008449673),new n.zr(11,18.014081827),new n.zr(12,19.017026896),new n.zr(13,20.023367295),new n.zr(14,21.027087574),new n.zr(15,22.034440259),new n.zr(16,23.04051),new n.zr(17,24.0505)]),F(k(l.I7,1),qt,3,0,[new n.zr(4,12.034404776),new n.zr(5,13.0248104),new n.zr(6,14.008595285),new n.zr(7,15.003065386),new n.zr(8,15.994914622),new n.zr(9,16.999131501),new n.zr(10,17.999160419),new n.zr(11,19.00357873),new n.zr(12,20.00407615),new n.zr(13,21.008654631),new n.zr(14,22.009967157),new n.zr(15,23.015691325),new n.zr(16,24.020369922),new n.zr(17,25.02914),new n.zr(18,26.03775)]),F(k(l.I7,1),qt,3,0,[new n.zr(5,14.03608),new n.zr(6,15.018010856),new n.zr(7,16.01146573),new n.zr(8,17.002095238),new n.zr(9,18.000937667),new n.zr(10,18.998403205),new n.zr(11,19.999981324),new n.zr(12,20.999948921),new n.zr(13,22.00299925),new n.zr(14,23.003574385),new n.zr(15,24.008099371),new n.zr(16,25.012094963),new n.zr(17,26.019633157),new n.zr(18,27.026892316),new n.zr(19,28.03567),new n.zr(20,29.04326)]),F(k(l.I7,1),qt,3,0,[new n.zr(6,16.025756907),new n.zr(7,17.017697565),new n.zr(8,18.005697066),new n.zr(9,19.001879839),new n.zr(10,19.992440176),new n.zr(11,20.993846744),new n.zr(12,21.99138551),new n.zr(13,22.994467337),new n.zr(14,23.993615074),new n.zr(15,24.997789899),new n.zr(16,26.000461498),new n.zr(17,27.0076152),new n.zr(18,28.012108072),new n.zr(19,29.019345902),new n.zr(20,30.023872),new n.zr(21,31.03311),new n.zr(22,32.03991)]),F(k(l.I7,1),qt,3,0,[new n.zr(7,18.02718),new n.zr(8,19.01387945),new n.zr(9,20.00734826),new n.zr(10,20.997655099),new n.zr(11,21.994436782),new n.zr(12,22.989769675),new n.zr(13,23.990963332),new n.zr(14,24.989954352),new n.zr(15,25.992589898),new n.zr(16,26.994008702),new n.zr(17,27.99889041),new n.zr(18,29.002811301),new n.zr(19,30.009226487),new n.zr(20,31.013595108),new n.zr(21,32.019649792),new n.zr(22,33.027386),new n.zr(23,34.0349),new n.zr(24,35.04418)]),F(k(l.I7,1),qt,3,0,[new n.zr(8,20.018862744),new n.zr(9,21.011714174),new n.zr(10,21.999574055),new n.zr(11,22.99412485),new n.zr(12,23.985041898),new n.zr(13,24.985837023),new n.zr(14,25.98259304),new n.zr(15,26.984340742),new n.zr(16,27.983876703),new n.zr(17,28.988554743),new n.zr(18,29.990464529),new n.zr(19,30.996548459),new n.zr(20,31.999145889),new n.zr(21,33.005586975),new n.zr(22,34.00907244),new n.zr(23,35.018669),new n.zr(24,36.02245),new n.zr(25,37.03124)]),F(k(l.I7,1),qt,3,0,[new n.zr(8,21.02804),new n.zr(9,22.01952),new n.zr(10,23.0072649),new n.zr(11,23.999940911),new n.zr(12,24.990428555),new n.zr(13,25.986891659),new n.zr(14,26.981538441),new n.zr(15,27.981910184),new n.zr(16,28.980444848),new n.zr(17,29.982960304),new n.zr(18,30.983946023),new n.zr(19,31.988124379),new n.zr(20,32.990869587),new n.zr(21,33.996927255),new n.zr(22,34.99993765),new n.zr(23,36.006351501),new n.zr(24,37.01031),new n.zr(25,38.0169),new n.zr(26,39.0219)]),F(k(l.I7,1),qt,3,0,[new n.zr(8,22.03453),new n.zr(9,23.02552),new n.zr(10,24.011545711),new n.zr(11,25.00410664),new n.zr(12,25.992329935),new n.zr(13,26.986704764),new n.zr(14,27.976926533),new n.zr(15,28.976494719),new n.zr(16,29.973770218),new n.zr(17,30.975363275),new n.zr(18,31.974148129),new n.zr(19,32.97800052),new n.zr(20,33.978575745),new n.zr(21,34.984584158),new n.zr(22,35.986687363),new n.zr(23,36.99299599),new n.zr(24,37.99598),new n.zr(25,39.0023),new n.zr(26,40.0058),new n.zr(27,41.0127),new n.zr(28,42.0161)]),F(k(l.I7,1),qt,3,0,[new n.zr(9,24.03435),new n.zr(10,25.02026),new n.zr(11,26.01178),new n.zr(12,26.999191645),new n.zr(13,27.99231233),new n.zr(14,28.981801376),new n.zr(15,29.978313807),new n.zr(16,30.973761512),new n.zr(17,31.973907163),new n.zr(18,32.971725281),new n.zr(19,33.973636381),new n.zr(20,34.973314249),new n.zr(21,35.978259824),new n.zr(22,36.979608338),new n.zr(23,37.98447),new n.zr(24,38.98642),new n.zr(25,39.99105),new n.zr(26,40.9948),new n.zr(27,42.00009),new n.zr(28,43.00331),new n.zr(29,44.00988),new n.zr(30,45.01514),new n.zr(31,46.02383)]),F(k(l.I7,1),qt,3,0,[new n.zr(10,26.02788),new n.zr(11,27.018795),new n.zr(12,28.004372661),new n.zr(13,28.996608805),new n.zr(14,29.984902954),new n.zr(15,30.979554421),new n.zr(16,31.97207069),new n.zr(17,32.971458497),new n.zr(18,33.967866831),new n.zr(19,34.96903214),new n.zr(20,35.96708088),new n.zr(21,36.971125716),new n.zr(22,37.971163443),new n.zr(23,38.975135275),new n.zr(24,39.97547),new n.zr(25,40.98003),new n.zr(26,41.98149),new n.zr(27,42.9866),new n.zr(28,43.98832),new n.zr(29,44.99482),new n.zr(30,45.99957),new n.zr(31,47.00762),new n.zr(32,48.01299),new n.zr(33,49.02201)]),F(k(l.I7,1),qt,3,0,[new n.zr(11,28.02851),new n.zr(12,29.01411),new n.zr(13,30.00477),new n.zr(14,30.992416014),new n.zr(15,31.985688908),new n.zr(16,32.977451798),new n.zr(17,33.973761967),new n.zr(18,34.968852707),new n.zr(19,35.968306945),new n.zr(20,36.9659026),new n.zr(21,37.96801055),new n.zr(22,38.968007677),new n.zr(23,39.970415555),new n.zr(24,40.970650212),new n.zr(25,41.973174994),new n.zr(26,42.974203385),new n.zr(27,43.978538712),new n.zr(28,44.9797),new n.zr(29,45.98412),new n.zr(30,46.98795),new n.zr(31,47.99485),new n.zr(32,48.99989),new n.zr(33,50.00773),new n.zr(34,51.01353)]),F(k(l.I7,1),qt,3,0,[new n.zr(12,30.02156),new n.zr(13,31.012126),new n.zr(14,31.99766066),new n.zr(15,32.989928719),new n.zr(16,33.980270118),new n.zr(17,34.975256726),new n.zr(18,35.967546282),new n.zr(19,36.966775912),new n.zr(20,37.962732161),new n.zr(21,38.964313413),new n.zr(22,39.962383123),new n.zr(23,40.964500828),new n.zr(24,41.963046386),new n.zr(25,42.965670701),new n.zr(26,43.965365269),new n.zr(27,44.968094979),new n.zr(28,45.968093467),new n.zr(29,46.972186238),new n.zr(30,47.97507),new n.zr(31,48.98218),new n.zr(32,49.98594),new n.zr(33,50.99324),new n.zr(34,51.99817),new n.zr(35,53.006227)]),F(k(l.I7,1),qt,3,0,[new n.zr(13,32.02192),new n.zr(14,33.00726),new n.zr(15,33.99841),new n.zr(16,34.988011615),new n.zr(17,35.981293405),new n.zr(18,36.973376915),new n.zr(19,37.969080107),new n.zr(20,38.963706861),new n.zr(21,39.963998672),new n.zr(22,40.961825972),new n.zr(23,41.962403059),new n.zr(24,42.960715746),new n.zr(25,43.961556146),new n.zr(26,44.960699658),new n.zr(27,45.961976203),new n.zr(28,46.961677807),new n.zr(29,47.965512946),new n.zr(30,48.967450084),new n.zr(31,49.972782832),new n.zr(32,50.97638),new n.zr(33,51.98261),new n.zr(34,52.98712),new n.zr(35,53.99399),new n.zr(36,54.999388)]),F(k(l.I7,1),qt,3,0,[new n.zr(14,34.01412),new n.zr(15,35.004765),new n.zr(16,35.993087234),new n.zr(17,36.985871505),new n.zr(18,37.976318637),new n.zr(19,38.970717729),new n.zr(20,39.962591155),new n.zr(21,40.962278349),new n.zr(22,41.958618337),new n.zr(23,42.958766833),new n.zr(24,43.955481094),new n.zr(25,44.956185938),new n.zr(26,45.953692759),new n.zr(27,46.954546459),new n.zr(28,47.952533512),new n.zr(29,48.955673302),new n.zr(30,49.957518286),new n.zr(31,50.961474238),new n.zr(32,51.9651),new n.zr(33,52.97005),new n.zr(34,53.97468),new n.zr(35,54.98055),new n.zr(36,55.98579),new n.zr(37,56.992356)]),F(k(l.I7,1),qt,3,0,[new n.zr(15,36.01492),new n.zr(16,37.00305),new n.zr(17,37.9947),new n.zr(18,38.984790009),new n.zr(19,39.977964014),new n.zr(20,40.969251316),new n.zr(21,41.965516761),new n.zr(22,42.96115098),new n.zr(23,43.959403048),new n.zr(24,44.955910243),new n.zr(25,45.95517025),new n.zr(26,46.952408027),new n.zr(27,47.952234991),new n.zr(28,48.950024065),new n.zr(29,49.952187008),new n.zr(30,50.9536027),new n.zr(31,51.95665),new n.zr(32,52.95817),new n.zr(33,53.963),new n.zr(34,54.9694),new n.zr(35,55.97266),new n.zr(36,56.97704),new n.zr(37,57.98307),new n.zr(38,58.988041)]),F(k(l.I7,1),qt,3,0,[new n.zr(16,38.00977),new n.zr(17,39.001323),new n.zr(18,39.990498907),new n.zr(19,40.983131),new n.zr(20,41.973031622),new n.zr(21,42.968523342),new n.zr(22,43.959690235),new n.zr(23,44.958124349),new n.zr(24,45.952629491),new n.zr(25,46.951763792),new n.zr(26,47.947947053),new n.zr(27,48.947870789),new n.zr(28,49.944792069),new n.zr(29,50.946616017),new n.zr(30,51.946898175),new n.zr(31,52.949731709),new n.zr(32,53.95087),new n.zr(33,54.95512),new n.zr(34,55.95799),new n.zr(35,56.9643),new n.zr(36,57.96611),new n.zr(37,58.97196),new n.zr(38,59.97564),new n.zr(39,60.982018)]),F(k(l.I7,1),qt,3,0,[new n.zr(17,40.01109),new n.zr(18,40.99974),new n.zr(19,41.99123),new n.zr(20,42.98065),new n.zr(21,43.9744),new n.zr(22,44.965782286),new n.zr(23,45.960199491),new n.zr(24,46.954906918),new n.zr(25,47.95225448),new n.zr(26,48.948516914),new n.zr(27,49.947162792),new n.zr(28,50.943963675),new n.zr(29,51.944779658),new n.zr(30,52.944342517),new n.zr(31,53.946444381),new n.zr(32,54.947238194),new n.zr(33,55.95036),new n.zr(34,56.95236),new n.zr(35,57.95665),new n.zr(36,58.9593),new n.zr(37,59.9645),new n.zr(38,60.96741),new n.zr(39,61.97314),new n.zr(40,62.97675)]),F(k(l.I7,1),qt,3,0,[new n.zr(18,42.00643),new n.zr(19,42.997707),new n.zr(20,43.98547),new n.zr(21,44.97916),new n.zr(22,45.968361649),new n.zr(23,46.962906512),new n.zr(24,47.954035861),new n.zr(25,48.951341135),new n.zr(26,49.946049607),new n.zr(27,50.944771767),new n.zr(28,51.940511904),new n.zr(29,52.940653781),new n.zr(30,53.938884921),new n.zr(31,54.940844164),new n.zr(32,55.940645238),new n.zr(33,56.9437538),new n.zr(34,57.94425),new n.zr(35,58.94863),new n.zr(36,59.94973),new n.zr(37,60.95409),new n.zr(38,61.9558),new n.zr(39,62.96186),new n.zr(40,63.9642),new n.zr(41,64.97037)]),F(k(l.I7,1),qt,3,0,[new n.zr(19,44.00687),new n.zr(20,44.99451),new n.zr(21,45.98672),new n.zr(22,46.9761),new n.zr(23,47.96887),new n.zr(24,48.959623415),new n.zr(25,49.95424396),new n.zr(26,50.948215487),new n.zr(27,51.945570079),new n.zr(28,52.941294702),new n.zr(29,53.940363247),new n.zr(30,54.938049636),new n.zr(31,55.938909366),new n.zr(32,56.938287458),new n.zr(33,57.939986451),new n.zr(34,58.940447166),new n.zr(35,59.943193998),new n.zr(36,60.94446),new n.zr(37,61.94797),new n.zr(38,62.94981),new n.zr(39,63.95373),new n.zr(40,64.9561),new n.zr(41,65.96082),new n.zr(42,66.96382)]),F(k(l.I7,1),qt,3,0,[new n.zr(19,45.01456),new n.zr(20,46.00081),new n.zr(21,46.99289),new n.zr(22,47.98056),new n.zr(23,48.97361),new n.zr(24,49.962993316),new n.zr(25,50.956824936),new n.zr(26,51.948116526),new n.zr(27,52.945312282),new n.zr(28,53.939614836),new n.zr(29,54.938298029),new n.zr(30,55.934942133),new n.zr(31,56.935398707),new n.zr(32,57.933280458),new n.zr(33,58.934880493),new n.zr(34,59.934076943),new n.zr(35,60.936749461),new n.zr(36,61.936770495),new n.zr(37,62.940118442),new n.zr(38,63.94087),new n.zr(39,64.94494),new n.zr(40,65.94598),new n.zr(41,66.95),new n.zr(42,67.95251),new n.zr(43,68.9577)]),F(k(l.I7,1),qt,3,0,[new n.zr(21,48.00176),new n.zr(22,48.98972),new n.zr(23,49.98154),new n.zr(24,50.97072),new n.zr(25,51.96359),new n.zr(26,52.954224985),new n.zr(27,53.948464147),new n.zr(28,54.942003149),new n.zr(29,55.939843937),new n.zr(30,56.936296235),new n.zr(31,57.935757571),new n.zr(32,58.933200194),new n.zr(33,59.933822196),new n.zr(34,60.932479381),new n.zr(35,61.934054212),new n.zr(36,62.933615218),new n.zr(37,63.935813523),new n.zr(38,64.936484581),new n.zr(39,65.939825412),new n.zr(40,66.94061),new n.zr(41,67.94436),new n.zr(42,68.9452),new n.zr(43,69.94981),new n.zr(44,70.95173),new n.zr(45,71.95641)]),F(k(l.I7,1),qt,3,0,[new n.zr(22,49.99593),new n.zr(23,50.98772),new n.zr(24,51.97568),new n.zr(25,52.96846),new n.zr(26,53.957910508),new n.zr(27,54.951336329),new n.zr(28,55.942136339),new n.zr(29,56.939800489),new n.zr(30,57.935347922),new n.zr(31,58.934351553),new n.zr(32,59.930790633),new n.zr(33,60.931060442),new n.zr(34,61.928348763),new n.zr(35,62.929672948),new n.zr(36,63.927969574),new n.zr(37,64.930088013),new n.zr(38,65.929115232),new n.zr(39,66.931569638),new n.zr(40,67.931844932),new n.zr(41,68.935181837),new n.zr(42,69.93614),new n.zr(43,70.94),new n.zr(44,71.9413),new n.zr(45,72.94608),new n.zr(46,73.94791),new n.zr(47,74.95297),new n.zr(48,75.95533),new n.zr(49,76.96083),new n.zr(50,77.9638)]),F(k(l.I7,1),qt,3,0,[new n.zr(23,51.99718),new n.zr(24,52.98555),new n.zr(25,53.97671),new n.zr(26,54.96605),new n.zr(27,55.95856),new n.zr(28,56.949215695),new n.zr(29,57.944540734),new n.zr(30,58.939504114),new n.zr(31,59.937368123),new n.zr(32,60.933462181),new n.zr(33,61.932587299),new n.zr(34,62.929601079),new n.zr(35,63.929767865),new n.zr(36,64.927793707),new n.zr(37,65.928873041),new n.zr(38,66.927750294),new n.zr(39,67.929637875),new n.zr(40,68.929425281),new n.zr(41,69.932409287),new n.zr(42,70.932619818),new n.zr(43,71.93552),new n.zr(44,72.93649),new n.zr(45,73.9402),new n.zr(46,74.9417),new n.zr(47,75.94599),new n.zr(48,76.94795),new n.zr(49,77.95281),new n.zr(50,78.95528),new n.zr(51,79.96189)]),F(k(l.I7,1),qt,3,0,[new n.zr(24,53.99295),new n.zr(25,54.98398),new n.zr(26,55.97238),new n.zr(27,56.96491),new n.zr(28,57.954596465),new n.zr(29,58.949267074),new n.zr(30,59.941832031),new n.zr(31,60.939513907),new n.zr(32,61.934334132),new n.zr(33,62.933215563),new n.zr(34,63.929146578),new n.zr(35,64.929245079),new n.zr(36,65.926036763),new n.zr(37,66.927130859),new n.zr(38,67.924847566),new n.zr(39,68.926553538),new n.zr(40,69.92532487),new n.zr(41,70.927727195),new n.zr(42,71.926861122),new n.zr(43,72.929779469),new n.zr(44,73.929458261),new n.zr(45,74.932937379),new n.zr(46,75.933394207),new n.zr(47,76.937085857),new n.zr(48,77.938569576),new n.zr(49,78.942095175),new n.zr(50,79.944414722),new n.zr(51,80.95048),new n.zr(52,81.95484)]),F(k(l.I7,1),qt,3,0,[new n.zr(25,55.99491),new n.zr(26,56.98293),new n.zr(27,57.97425),new n.zr(28,58.96337),new n.zr(29,59.95706),new n.zr(30,60.94917),new n.zr(31,61.944179608),new n.zr(32,62.939141527),new n.zr(33,63.936838307),new n.zr(34,64.932739322),new n.zr(35,65.931592355),new n.zr(36,66.928204915),new n.zr(37,67.927983497),new n.zr(38,68.925580912),new n.zr(39,69.926027741),new n.zr(40,70.92470501),new n.zr(41,71.92636935),new n.zr(42,72.925169832),new n.zr(43,73.926940999),new n.zr(44,74.926500645),new n.zr(45,75.928928262),new n.zr(46,76.929281189),new n.zr(47,77.93165595),new n.zr(48,78.932916371),new n.zr(49,79.936588154),new n.zr(50,80.937752955),new n.zr(51,81.94316),new n.zr(52,82.94687),new n.zr(53,83.95234)]),F(k(l.I7,1),qt,3,0,[new n.zr(26,57.99101),new n.zr(27,58.98175),new n.zr(28,59.97019),new n.zr(29,60.96379),new n.zr(30,61.95465),new n.zr(31,62.94964),new n.zr(32,63.941572638),new n.zr(33,64.939440762),new n.zr(34,65.933846798),new n.zr(35,66.932738415),new n.zr(36,67.928097266),new n.zr(37,68.927972002),new n.zr(38,69.924250365),new n.zr(39,70.924953991),new n.zr(40,71.922076184),new n.zr(41,72.923459361),new n.zr(42,73.921178213),new n.zr(43,74.922859494),new n.zr(44,75.921402716),new n.zr(45,76.923548462),new n.zr(46,77.922852886),new n.zr(47,78.92540156),new n.zr(48,79.925444764),new n.zr(49,80.928821065),new n.zr(50,81.929550326),new n.zr(51,82.93451),new n.zr(52,83.93731),new n.zr(53,84.94269),new n.zr(54,85.94627)]),F(k(l.I7,1),qt,3,0,[new n.zr(27,59.99313),new n.zr(28,60.98062),new n.zr(29,61.9732),new n.zr(30,62.96369),new n.zr(31,63.957572),new n.zr(32,64.949484),new n.zr(33,65.944099147),new n.zr(34,66.939190417),new n.zr(35,67.936792976),new n.zr(36,68.932280154),new n.zr(37,69.930927811),new n.zr(38,70.927114724),new n.zr(39,71.926752647),new n.zr(40,72.923825288),new n.zr(41,73.923929076),new n.zr(42,74.921596417),new n.zr(43,75.922393933),new n.zr(44,76.920647703),new n.zr(45,77.921828577),new n.zr(46,78.920948498),new n.zr(47,79.922578162),new n.zr(48,80.922132884),new n.zr(49,81.924504668),new n.zr(50,82.924980625),new n.zr(51,83.92906),new n.zr(52,84.93181),new n.zr(53,85.93623),new n.zr(54,86.93958),new n.zr(55,87.94456),new n.zr(56,88.94923)]),F(k(l.I7,1),qt,3,0,[new n.zr(31,64.96466),new n.zr(32,65.95521),new n.zr(33,66.95009),new n.zr(34,67.94187),new n.zr(35,68.939562155),new n.zr(36,69.933504),new n.zr(37,70.931868378),new n.zr(38,71.927112313),new n.zr(39,72.9267668),new n.zr(40,73.922476561),new n.zr(41,74.922523571),new n.zr(42,75.919214107),new n.zr(43,76.91991461),new n.zr(44,77.917309522),new n.zr(45,78.918499802),new n.zr(46,79.916521828),new n.zr(47,80.917992931),new n.zr(48,81.9167),new n.zr(49,82.919119072),new n.zr(50,83.918464523),new n.zr(51,84.922244678),new n.zr(52,85.924271165),new n.zr(53,86.928520749),new n.zr(54,87.931423982),new n.zr(55,88.93602),new n.zr(56,89.93942),new n.zr(57,90.94537),new n.zr(58,91.94933)]),F(k(l.I7,1),qt,3,0,[new n.zr(32,66.96479),new n.zr(33,67.958248),new n.zr(34,68.950178),new n.zr(35,69.944208),new n.zr(36,70.939246),new n.zr(37,71.936496876),new n.zr(38,72.931794889),new n.zr(39,73.929891152),new n.zr(40,74.92577641),new n.zr(41,75.924541974),new n.zr(42,76.921380123),new n.zr(43,77.92114613),new n.zr(44,78.918337647),new n.zr(45,79.918529952),new n.zr(46,80.91629106),new n.zr(47,81.916804666),new n.zr(48,82.915180219),new n.zr(49,83.916503685),new n.zr(50,84.915608027),new n.zr(51,85.918797162),new n.zr(52,86.920710713),new n.zr(53,87.924065908),new n.zr(54,88.92638726),new n.zr(55,89.930634988),new n.zr(56,90.9339653),new n.zr(57,91.939255258),new n.zr(58,92.9431),new n.zr(59,93.94868)]),F(k(l.I7,1),qt,3,0,[new n.zr(33,68.96532),new n.zr(34,69.95601),new n.zr(35,70.95051),new n.zr(36,71.94190754),new n.zr(37,72.938931115),new n.zr(38,73.933258225),new n.zr(39,74.931033794),new n.zr(40,75.925948304),new n.zr(41,76.92466788),new n.zr(42,77.920386271),new n.zr(43,78.920082992),new n.zr(44,79.91637804),new n.zr(45,80.916592419),new n.zr(46,81.913484601),new n.zr(47,82.914135952),new n.zr(48,83.911506627),new n.zr(49,84.912526954),new n.zr(50,85.910610313),new n.zr(51,86.913354251),new n.zr(52,87.914446951),new n.zr(53,88.917632505),new n.zr(54,89.919523803),new n.zr(55,90.923442418),new n.zr(56,91.926152752),new n.zr(57,92.931265246),new n.zr(58,93.934362),new n.zr(59,94.93984),new n.zr(60,95.94307),new n.zr(61,96.94856)]),F(k(l.I7,1),qt,3,0,[new n.zr(34,70.96532),new n.zr(35,71.95908),new n.zr(36,72.950366),new n.zr(37,73.944470376),new n.zr(38,74.938569199),new n.zr(39,75.935071448),new n.zr(40,76.930406599),new n.zr(41,77.928141485),new n.zr(42,78.923996719),new n.zr(43,79.922519322),new n.zr(44,80.918994165),new n.zr(45,81.918207691),new n.zr(46,82.915111951),new n.zr(47,83.914384676),new n.zr(48,84.911789341),new n.zr(49,85.91116708),new n.zr(50,86.909183465),new n.zr(51,87.911318556),new n.zr(52,88.912279939),new n.zr(53,89.914808941),new n.zr(54,90.91653416),new n.zr(55,91.919725442),new n.zr(56,92.922032765),new n.zr(57,93.926407326),new n.zr(58,94.92931926),new n.zr(59,95.934283962),new n.zr(60,96.937342863),new n.zr(61,97.941703557),new n.zr(62,98.945420616),new n.zr(63,99.94987),new n.zr(64,100.953195994),new n.zr(65,101.95921)]),F(k(l.I7,1),qt,3,0,[new n.zr(35,72.96597),new n.zr(36,73.95631),new n.zr(37,74.94992),new n.zr(38,75.94161),new n.zr(39,76.937761511),new n.zr(40,77.932179362),new n.zr(41,78.929707076),new n.zr(42,79.924524588),new n.zr(43,80.923213095),new n.zr(44,81.918401258),new n.zr(45,82.917555029),new n.zr(46,83.913424778),new n.zr(47,84.912932689),new n.zr(48,85.909262351),new n.zr(49,86.908879316),new n.zr(50,87.905614339),new n.zr(51,88.907452906),new n.zr(52,89.907737596),new n.zr(53,90.910209845),new n.zr(54,91.911029895),new n.zr(55,92.91402241),new n.zr(56,93.915359856),new n.zr(57,94.919358213),new n.zr(58,95.921680473),new n.zr(59,96.926148757),new n.zr(60,97.928471177),new n.zr(61,98.933315038),new n.zr(62,99.935351729),new n.zr(63,100.940517434),new n.zr(64,101.943018795),new n.zr(65,102.94895),new n.zr(66,103.95233)]),F(k(l.I7,1),qt,3,0,[new n.zr(38,76.94962),new n.zr(39,77.9435),new n.zr(40,78.937350712),new n.zr(41,79.931982402),new n.zr(42,80.929128719),new n.zr(43,81.926792071),new n.zr(44,82.922352572),new n.zr(45,83.920387768),new n.zr(46,84.916427076),new n.zr(47,85.914887724),new n.zr(48,86.910877833),new n.zr(49,87.909503361),new n.zr(50,88.905847902),new n.zr(51,89.907151443),new n.zr(52,90.907303415),new n.zr(53,91.908946832),new n.zr(54,92.909581582),new n.zr(55,93.911594008),new n.zr(56,94.912823709),new n.zr(57,95.915897787),new n.zr(58,96.918131017),new n.zr(59,97.922219525),new n.zr(60,98.924634736),new n.zr(61,99.927756402),new n.zr(62,100.930313395),new n.zr(63,101.933555501),new n.zr(64,102.93694),new n.zr(65,103.94145),new n.zr(66,104.94509),new n.zr(67,105.95022)]),F(k(l.I7,1),qt,3,0,[new n.zr(39,78.94916),new n.zr(40,79.94055),new n.zr(41,80.936815296),new n.zr(42,81.931086249),new n.zr(43,82.92865213),new n.zr(44,83.92325),new n.zr(45,84.92146522),new n.zr(46,85.916472851),new n.zr(47,86.914816578),new n.zr(48,87.910226179),new n.zr(49,88.908888916),new n.zr(50,89.904703679),new n.zr(51,90.905644968),new n.zr(52,91.905040106),new n.zr(53,92.906475627),new n.zr(54,93.906315765),new n.zr(55,94.908042739),new n.zr(56,95.908275675),new n.zr(57,96.910950716),new n.zr(58,97.912746366),new n.zr(59,98.916511084),new n.zr(60,99.917761704),new n.zr(61,100.921139958),new n.zr(62,101.922981089),new n.zr(63,102.926597062),new n.zr(64,103.92878),new n.zr(65,104.93305),new n.zr(66,105.93591),new n.zr(67,106.94086),new n.zr(68,107.94428)]),F(k(l.I7,1),qt,3,0,[new n.zr(40,80.94905),new n.zr(41,81.94313),new n.zr(42,82.936703713),new n.zr(43,83.93357),new n.zr(44,84.927906486),new n.zr(45,85.925037588),new n.zr(46,86.920361435),new n.zr(47,87.91833144),new n.zr(48,88.913495503),new n.zr(49,89.911264109),new n.zr(50,90.906990538),new n.zr(51,91.907193214),new n.zr(52,92.906377543),new n.zr(53,93.907283457),new n.zr(54,94.906835178),new n.zr(55,95.908100076),new n.zr(56,96.908097144),new n.zr(57,97.91033069),new n.zr(58,98.911617864),new n.zr(59,99.914181434),new n.zr(60,100.915251567),new n.zr(61,101.918037417),new n.zr(62,102.919141297),new n.zr(63,103.922459464),new n.zr(64,104.923934023),new n.zr(65,105.92819),new n.zr(66,106.93031),new n.zr(67,107.93501),new n.zr(68,108.93763),new n.zr(69,109.94268)]),F(k(l.I7,1),qt,3,0,[new n.zr(41,82.94874),new n.zr(42,83.94009),new n.zr(43,84.93659),new n.zr(44,85.930695167),new n.zr(45,86.92732683),new n.zr(46,87.921952728),new n.zr(47,88.919480562),new n.zr(48,89.913936161),new n.zr(49,90.911750754),new n.zr(50,91.90681048),new n.zr(51,92.906812213),new n.zr(52,93.905087578),new n.zr(53,94.905841487),new n.zr(54,95.904678904),new n.zr(55,96.906021033),new n.zr(56,97.905407846),new n.zr(57,98.907711598),new n.zr(58,99.907477149),new n.zr(59,100.910346543),new n.zr(60,101.910297162),new n.zr(61,102.913204596),new n.zr(62,103.913758387),new n.zr(63,104.916972087),new n.zr(64,105.918134284),new n.zr(65,106.921694724),new n.zr(66,107.923973837),new n.zr(67,108.92781),new n.zr(68,109.92973),new n.zr(69,110.93451),new n.zr(70,111.93684),new n.zr(71,112.94203)]),F(k(l.I7,1),qt,3,0,[new n.zr(42,84.94894),new n.zr(43,85.94288),new n.zr(44,86.93653),new n.zr(45,87.93283),new n.zr(46,88.92754288),new n.zr(47,89.92355583),new n.zr(48,90.9184282),new n.zr(49,91.915259655),new n.zr(50,92.910248473),new n.zr(51,93.909656309),new n.zr(52,94.907656454),new n.zr(53,95.907870803),new n.zr(54,96.906364843),new n.zr(55,97.907215692),new n.zr(56,98.906254554),new n.zr(57,99.907657594),new n.zr(58,100.90731438),new n.zr(59,101.909212938),new n.zr(60,102.909178805),new n.zr(61,103.911444898),new n.zr(62,104.911658043),new n.zr(63,105.914355408),new n.zr(64,106.915081691),new n.zr(65,107.918479973),new n.zr(66,108.919980998),new n.zr(67,109.92339),new n.zr(68,110.92505),new n.zr(69,111.92924),new n.zr(70,112.93133),new n.zr(71,113.93588),new n.zr(72,114.93828)]),F(k(l.I7,1),qt,3,0,[new n.zr(43,86.94918),new n.zr(44,87.94042),new n.zr(45,88.93611),new n.zr(46,89.92978),new n.zr(47,90.926377434),new n.zr(48,91.92012),new n.zr(49,92.917051523),new n.zr(50,93.911359569),new n.zr(51,94.910412729),new n.zr(52,95.907597681),new n.zr(53,96.907554546),new n.zr(54,97.905287111),new n.zr(55,98.905939307),new n.zr(56,99.904219664),new n.zr(57,100.905582219),new n.zr(58,101.904349503),new n.zr(59,102.906323677),new n.zr(60,103.905430145),new n.zr(61,104.907750341),new n.zr(62,105.907326913),new n.zr(63,106.909907207),new n.zr(64,107.910192211),new n.zr(65,108.913201565),new n.zr(66,109.913966185),new n.zr(67,110.91756),new n.zr(68,111.918821673),new n.zr(69,112.92254),new n.zr(70,113.923891981),new n.zr(71,114.92831),new n.zr(72,115.93016),new n.zr(73,116.93479),new n.zr(74,117.93703)]),F(k(l.I7,1),qt,3,0,[new n.zr(44,88.94938),new n.zr(45,89.94287),new n.zr(46,90.93655),new n.zr(47,91.93198),new n.zr(48,92.92574),new n.zr(49,93.921698),new n.zr(50,94.915898541),new n.zr(51,95.914518212),new n.zr(52,96.911336643),new n.zr(53,97.910716431),new n.zr(54,98.908132101),new n.zr(55,99.90811663),new n.zr(56,100.906163526),new n.zr(57,101.906842845),new n.zr(58,102.905504182),new n.zr(59,103.906655315),new n.zr(60,104.905692444),new n.zr(61,105.907284615),new n.zr(62,106.90675054),new n.zr(63,107.908730768),new n.zr(64,108.908735621),new n.zr(65,109.910949525),new n.zr(66,110.91166),new n.zr(67,111.913969253),new n.zr(68,112.91542),new n.zr(69,113.91734336),new n.zr(70,114.920124676),new n.zr(71,115.922746643),new n.zr(72,116.92535),new n.zr(73,117.92943),new n.zr(74,118.93136),new n.zr(75,119.93578),new n.zr(76,120.93808)]),F(k(l.I7,1),qt,3,0,[new n.zr(45,90.94948),new n.zr(46,91.94042),new n.zr(47,92.93591),new n.zr(48,93.92877),new n.zr(49,94.92469),new n.zr(50,95.91822194),new n.zr(51,96.916478921),new n.zr(52,97.912720751),new n.zr(53,98.911767757),new n.zr(54,99.908504596),new n.zr(55,100.908289144),new n.zr(56,101.905607716),new n.zr(57,102.906087204),new n.zr(58,103.904034912),new n.zr(59,104.905084046),new n.zr(60,105.903483087),new n.zr(61,106.905128453),new n.zr(62,107.903894451),new n.zr(63,108.905953535),new n.zr(64,109.905152385),new n.zr(65,110.907643952),new n.zr(66,111.907313277),new n.zr(67,112.910151346),new n.zr(68,113.910365322),new n.zr(69,114.91368341),new n.zr(70,115.914158288),new n.zr(71,116.91784),new n.zr(72,117.918983915),new n.zr(73,118.92268),new n.zr(74,119.92403),new n.zr(75,120.92818),new n.zr(76,121.9298),new n.zr(77,122.93426)]),F(k(l.I7,1),qt,3,0,[new n.zr(47,93.94278),new n.zr(48,94.93548),new n.zr(49,95.93068),new n.zr(50,96.924),new n.zr(51,97.921759995),new n.zr(52,98.917597103),new n.zr(53,99.916069387),new n.zr(54,100.912802135),new n.zr(55,101.911999996),new n.zr(56,102.908972453),new n.zr(57,103.908628228),new n.zr(58,104.906528234),new n.zr(59,105.906666431),new n.zr(60,106.90509302),new n.zr(61,107.905953705),new n.zr(62,108.904755514),new n.zr(63,109.90611046),new n.zr(64,110.905294679),new n.zr(65,111.907004132),new n.zr(66,112.906565708),new n.zr(67,113.908807907),new n.zr(68,114.908762282),new n.zr(69,115.911359558),new n.zr(70,116.911684187),new n.zr(71,117.914582383),new n.zr(72,118.915666045),new n.zr(73,119.918788609),new n.zr(74,120.919851074),new n.zr(75,121.92332),new n.zr(76,122.9249),new n.zr(77,123.92853),new n.zr(78,124.93054),new n.zr(79,125.9345),new n.zr(80,126.93688)]),F(k(l.I7,1),qt,3,0,[new n.zr(48,95.93977),new n.zr(49,96.93494),new n.zr(50,97.927579),new n.zr(51,98.92501),new n.zr(52,99.920230232),new n.zr(53,100.918681442),new n.zr(54,101.914777255),new n.zr(55,102.913418952),new n.zr(56,103.909848091),new n.zr(57,104.909467818),new n.zr(58,105.906458007),new n.zr(59,106.906614232),new n.zr(60,107.904183403),new n.zr(61,108.904985569),new n.zr(62,109.903005578),new n.zr(63,110.904181628),new n.zr(64,111.902757226),new n.zr(65,112.904400947),new n.zr(66,113.903358121),new n.zr(67,114.905430553),new n.zr(68,115.904755434),new n.zr(69,116.907218242),new n.zr(70,117.906914144),new n.zr(71,118.909922582),new n.zr(72,119.909851352),new n.zr(73,120.91298039),new n.zr(74,121.9135),new n.zr(75,122.917003675),new n.zr(76,123.917648302),new n.zr(77,124.92124717),new n.zr(78,125.922353996),new n.zr(79,126.926434822),new n.zr(80,127.927760617),new n.zr(81,128.93226),new n.zr(82,129.93398)]),F(k(l.I7,1),qt,3,0,[new n.zr(49,97.94224),new n.zr(50,98.93461),new n.zr(51,99.931149033),new n.zr(52,100.92656),new n.zr(53,101.924707541),new n.zr(54,102.919913896),new n.zr(55,103.918338416),new n.zr(56,104.914673434),new n.zr(57,105.913461134),new n.zr(58,106.910292195),new n.zr(59,107.909719683),new n.zr(60,108.907154078),new n.zr(61,109.907168783),new n.zr(62,110.905110677),new n.zr(63,111.905533338),new n.zr(64,112.904061223),new n.zr(65,113.904916758),new n.zr(66,114.903878328),new n.zr(67,115.905259995),new n.zr(68,116.904515731),new n.zr(69,117.906354623),new n.zr(70,118.905846334),new n.zr(71,119.907961505),new n.zr(72,120.907848847),new n.zr(73,121.910277103),new n.zr(74,122.910438951),new n.zr(75,123.913175916),new n.zr(76,124.913601387),new n.zr(77,125.916464532),new n.zr(78,126.917344048),new n.zr(79,127.920170658),new n.zr(80,128.921657958),new n.zr(81,129.924854941),new n.zr(82,130.926767408),new n.zr(83,131.932919005),new n.zr(84,132.93834),new n.zr(85,133.94466)]),F(k(l.I7,1),qt,3,0,[new n.zr(50,99.938954),new n.zr(51,100.93606),new n.zr(52,101.93049),new n.zr(53,102.92813),new n.zr(54,103.923185469),new n.zr(55,104.921390409),new n.zr(56,105.916880472),new n.zr(57,106.915666702),new n.zr(58,107.911965339),new n.zr(59,108.911286879),new n.zr(60,109.907852688),new n.zr(61,110.907735404),new n.zr(62,111.90482081),new n.zr(63,112.905173373),new n.zr(64,113.902781816),new n.zr(65,114.903345973),new n.zr(66,115.901744149),new n.zr(67,116.902953765),new n.zr(68,117.901606328),new n.zr(69,118.90330888),new n.zr(70,119.902196571),new n.zr(71,120.904236867),new n.zr(72,121.903440138),new n.zr(73,122.905721901),new n.zr(74,123.90527463),new n.zr(75,124.907784924),new n.zr(76,125.907653953),new n.zr(77,126.91035098),new n.zr(78,127.910534953),new n.zr(79,128.913439976),new n.zr(80,129.913852185),new n.zr(81,130.916919144),new n.zr(82,131.917744455),new n.zr(83,132.923814085),new n.zr(84,133.928463576),new n.zr(85,134.93473),new n.zr(86,135.93934),new n.zr(87,136.94579)]),F(k(l.I7,1),qt,3,0,[new n.zr(52,102.94012),new n.zr(53,103.936287),new n.zr(54,104.931528593),new n.zr(55,105.928183134),new n.zr(56,106.92415),new n.zr(57,107.92216),new n.zr(58,108.918136092),new n.zr(59,109.917533911),new n.zr(60,110.912534147),new n.zr(61,111.91239464),new n.zr(62,112.909377941),new n.zr(63,113.909095876),new n.zr(64,114.906598812),new n.zr(65,115.906797235),new n.zr(66,116.90483959),new n.zr(67,117.905531885),new n.zr(68,118.90394646),new n.zr(69,119.905074315),new n.zr(70,120.903818044),new n.zr(71,121.905175415),new n.zr(72,122.904215696),new n.zr(73,123.905937525),new n.zr(74,124.905247804),new n.zr(75,125.907248153),new n.zr(76,126.906914564),new n.zr(77,127.90916733),new n.zr(78,128.909150092),new n.zr(79,129.911546459),new n.zr(80,130.911946487),new n.zr(81,131.914413247),new n.zr(82,132.915236466),new n.zr(83,133.920551554),new n.zr(84,134.925167962),new n.zr(85,135.93066),new n.zr(86,136.93531),new n.zr(87,137.94096),new n.zr(88,138.94571)]),F(k(l.I7,1),qt,3,0,[new n.zr(54,105.937702),new n.zr(55,106.935036),new n.zr(56,107.929486838),new n.zr(57,108.927456483),new n.zr(58,109.922407164),new n.zr(59,110.921120589),new n.zr(60,111.917061617),new n.zr(61,112.915452551),new n.zr(62,113.912498025),new n.zr(63,114.911578627),new n.zr(64,115.908420253),new n.zr(65,116.90863418),new n.zr(66,117.905825187),new n.zr(67,118.90640811),new n.zr(68,119.904019891),new n.zr(69,120.904929815),new n.zr(70,121.903047064),new n.zr(71,122.904272951),new n.zr(72,123.902819466),new n.zr(73,124.904424718),new n.zr(74,125.903305543),new n.zr(75,126.90521729),new n.zr(76,127.904461383),new n.zr(77,128.906595593),new n.zr(78,129.906222753),new n.zr(79,130.90852188),new n.zr(80,131.908523782),new n.zr(81,132.910939068),new n.zr(82,133.911540546),new n.zr(83,134.916450782),new n.zr(84,135.920103155),new n.zr(85,136.925324769),new n.zr(86,137.92922),new n.zr(87,138.93473),new n.zr(88,139.9387),new n.zr(89,140.94439),new n.zr(90,141.9485)]),F(k(l.I7,1),qt,3,0,[new n.zr(55,107.943291),new n.zr(56,108.938191658),new n.zr(57,109.934634181),new n.zr(58,110.930276),new n.zr(59,111.92797),new n.zr(60,112.923644245),new n.zr(61,113.92185),new n.zr(62,114.918272),new n.zr(63,115.916735014),new n.zr(64,116.913647692),new n.zr(65,117.91337523),new n.zr(66,118.910180837),new n.zr(67,119.910047843),new n.zr(68,120.907366063),new n.zr(69,121.907592451),new n.zr(70,122.905597944),new n.zr(71,123.906211423),new n.zr(72,124.90462415),new n.zr(73,125.905619387),new n.zr(74,126.90446842),new n.zr(75,127.905805254),new n.zr(76,128.904987487),new n.zr(77,129.906674018),new n.zr(78,130.906124168),new n.zr(79,131.907994525),new n.zr(80,132.907806465),new n.zr(81,133.909876552),new n.zr(82,134.91005031),new n.zr(83,135.914655105),new n.zr(84,136.917872653),new n.zr(85,137.922383666),new n.zr(86,138.926093402),new n.zr(87,139.93121),new n.zr(88,140.93483),new n.zr(89,141.94018),new n.zr(90,142.94407),new n.zr(91,143.94961)]),F(k(l.I7,1),qt,3,0,[new n.zr(56,109.944476),new n.zr(57,110.941632),new n.zr(58,111.93566535),new n.zr(59,112.933382836),new n.zr(60,113.928145),new n.zr(61,114.926979032),new n.zr(62,115.921394197),new n.zr(63,116.920564355),new n.zr(64,117.91657092),new n.zr(65,118.915554295),new n.zr(66,119.91215199),new n.zr(67,120.911386497),new n.zr(68,121.908548396),new n.zr(69,122.908470748),new n.zr(70,123.905895774),new n.zr(71,124.906398236),new n.zr(72,125.904268868),new n.zr(73,126.905179581),new n.zr(74,127.903530436),new n.zr(75,128.904779458),new n.zr(76,129.903507903),new n.zr(77,130.90508192),new n.zr(78,131.904154457),new n.zr(79,132.90590566),new n.zr(80,133.905394504),new n.zr(81,134.907207499),new n.zr(82,135.907219526),new n.zr(83,136.911562939),new n.zr(84,137.913988549),new n.zr(85,138.918786859),new n.zr(86,139.921635665),new n.zr(87,140.926646282),new n.zr(88,141.929702981),new n.zr(89,142.93489),new n.zr(90,143.93823),new n.zr(91,144.94367),new n.zr(92,145.9473),new n.zr(93,146.95301)]),F(k(l.I7,1),qt,3,0,[new n.zr(57,111.950331),new n.zr(58,112.944535512),new n.zr(59,113.940841319),new n.zr(60,114.935939),new n.zr(61,115.932914152),new n.zr(62,116.928639484),new n.zr(63,117.926554883),new n.zr(64,118.922370879),new n.zr(65,119.920678219),new n.zr(66,120.917183637),new n.zr(67,121.916121946),new n.zr(68,122.912990168),new n.zr(69,123.912245731),new n.zr(70,124.909724871),new n.zr(71,125.909447953),new n.zr(72,126.9074176),new n.zr(73,127.907747919),new n.zr(74,128.906063369),new n.zr(75,129.906706163),new n.zr(76,130.905460232),new n.zr(77,131.906429799),new n.zr(78,132.90544687),new n.zr(79,133.906713419),new n.zr(80,134.905971903),new n.zr(81,135.907305741),new n.zr(82,136.907083505),new n.zr(83,137.911010537),new n.zr(84,138.913357921),new n.zr(85,139.917277075),new n.zr(86,140.920043984),new n.zr(87,141.924292317),new n.zr(88,142.927330292),new n.zr(89,143.932027373),new n.zr(90,144.935388226),new n.zr(91,145.940162028),new n.zr(92,146.943864435),new n.zr(93,147.948899539),new n.zr(94,148.95272),new n.zr(95,149.95797),new n.zr(96,150.962)]),F(k(l.I7,1),qt,3,0,[new n.zr(58,113.950941),new n.zr(59,114.94771),new n.zr(60,115.94168),new n.zr(61,116.937700229),new n.zr(62,117.93344),new n.zr(63,118.931051927),new n.zr(64,119.926045941),new n.zr(65,120.924485908),new n.zr(66,121.92026),new n.zr(67,122.91885),new n.zr(68,123.915088437),new n.zr(69,124.914620234),new n.zr(70,125.911244146),new n.zr(71,126.911121328),new n.zr(72,127.90830887),new n.zr(73,128.908673749),new n.zr(74,129.906310478),new n.zr(75,130.906930798),new n.zr(76,131.905056152),new n.zr(77,132.906002368),new n.zr(78,133.904503347),new n.zr(79,134.905682749),new n.zr(80,135.904570109),new n.zr(81,136.905821414),new n.zr(82,137.905241273),new n.zr(83,138.908835384),new n.zr(84,139.910599485),new n.zr(85,140.914406439),new n.zr(86,141.916448175),new n.zr(87,142.920617184),new n.zr(88,143.922940468),new n.zr(89,144.926923807),new n.zr(90,145.930106645),new n.zr(91,146.933992519),new n.zr(92,147.937682377),new n.zr(93,148.94246),new n.zr(94,149.94562),new n.zr(95,150.9507),new n.zr(96,151.95416),new n.zr(97,152.95961)]),F(k(l.I7,1),qt,3,0,[new n.zr(60,116.95001),new n.zr(61,117.94657),new n.zr(62,118.94099),new n.zr(63,119.93807),new n.zr(64,120.93301),new n.zr(65,121.93071),new n.zr(66,122.92624),new n.zr(67,123.92453),new n.zr(68,124.92067),new n.zr(69,125.91937),new n.zr(70,126.91616),new n.zr(71,127.91544794),new n.zr(72,128.912667334),new n.zr(73,129.91232),new n.zr(74,130.910108489),new n.zr(75,131.910110399),new n.zr(76,132.908396372),new n.zr(77,133.908489607),new n.zr(78,134.906971003),new n.zr(79,135.907651181),new n.zr(80,136.906465656),new n.zr(81,137.907106826),new n.zr(82,138.90634816),new n.zr(83,139.909472552),new n.zr(84,140.910957016),new n.zr(85,141.914074489),new n.zr(86,142.916058646),new n.zr(87,143.919591666),new n.zr(88,144.92163837),new n.zr(89,145.925700146),new n.zr(90,146.927819639),new n.zr(91,147.932191197),new n.zr(92,148.93437),new n.zr(93,149.93857),new n.zr(94,150.94156),new n.zr(95,151.94611),new n.zr(96,152.94945),new n.zr(97,153.9544),new n.zr(98,154.95813)]),F(k(l.I7,1),qt,3,0,[new n.zr(61,118.95276),new n.zr(62,119.94664),new n.zr(63,120.94367),new n.zr(64,121.93801),new n.zr(65,122.93551),new n.zr(66,123.93052),new n.zr(67,124.92854),new n.zr(68,125.9241),new n.zr(69,126.92275),new n.zr(70,127.91887),new n.zr(71,128.918679183),new n.zr(72,129.914339361),new n.zr(73,130.914424137),new n.zr(74,131.91149),new n.zr(75,132.91155),new n.zr(76,133.909026379),new n.zr(77,134.909145555),new n.zr(78,135.907143574),new n.zr(79,136.907777634),new n.zr(80,137.905985574),new n.zr(81,138.906646605),new n.zr(82,139.905434035),new n.zr(83,140.908271103),new n.zr(84,141.909239733),new n.zr(85,142.912381158),new n.zr(86,143.913642686),new n.zr(87,144.917227871),new n.zr(88,145.918689722),new n.zr(89,146.922510962),new n.zr(90,147.924394738),new n.zr(91,148.928289207),new n.zr(92,149.930226399),new n.zr(93,150.93404),new n.zr(94,151.93638),new n.zr(95,152.94058),new n.zr(96,153.94332),new n.zr(97,154.94804),new n.zr(98,155.95126),new n.zr(99,156.95634)]),F(k(l.I7,1),qt,3,0,[new n.zr(62,120.955364),new n.zr(63,121.95165),new n.zr(64,122.94596),new n.zr(65,123.94296),new n.zr(66,124.93783),new n.zr(67,125.93531),new n.zr(68,126.93083),new n.zr(69,127.9288),new n.zr(70,128.92486),new n.zr(71,129.92338),new n.zr(72,130.920060245),new n.zr(73,131.91912),new n.zr(74,132.9162),new n.zr(75,133.915672),new n.zr(76,134.91313914),new n.zr(77,135.912646935),new n.zr(78,136.910678351),new n.zr(79,137.910748891),new n.zr(80,138.908932181),new n.zr(81,139.909071204),new n.zr(82,140.907647726),new n.zr(83,141.910039865),new n.zr(84,142.910812233),new n.zr(85,143.913300595),new n.zr(86,144.914506897),new n.zr(87,145.917588016),new n.zr(88,146.918979001),new n.zr(89,147.922183237),new n.zr(90,148.923791056),new n.zr(91,149.926995031),new n.zr(92,150.928227869),new n.zr(93,151.9316),new n.zr(94,152.93365),new n.zr(95,153.93739),new n.zr(96,154.93999),new n.zr(97,155.94412),new n.zr(98,156.94717),new n.zr(99,157.95178),new n.zr(100,158.95523)]),F(k(l.I7,1),qt,3,0,[new n.zr(66,125.94307),new n.zr(67,126.9405),new n.zr(68,127.93539),new n.zr(69,128.932385),new n.zr(70,129.92878),new n.zr(71,130.927102697),new n.zr(72,131.92312),new n.zr(73,132.92221),new n.zr(74,133.918645),new n.zr(75,134.91824),new n.zr(76,135.915020542),new n.zr(77,136.91463973),new n.zr(78,137.91291745),new n.zr(79,138.91192415),new n.zr(80,139.909309824),new n.zr(81,140.9096048),new n.zr(82,141.907718643),new n.zr(83,142.909809626),new n.zr(84,143.910082629),new n.zr(85,144.912568847),new n.zr(86,145.913112139),new n.zr(87,146.916095794),new n.zr(88,147.916888516),new n.zr(89,148.92014419),new n.zr(90,149.920886563),new n.zr(91,150.923824739),new n.zr(92,151.924682428),new n.zr(93,152.927694534),new n.zr(94,153.929483295),new n.zr(95,154.932629551),new n.zr(96,155.9352),new n.zr(97,156.93927),new n.zr(98,157.94187),new n.zr(99,158.94639),new n.zr(100,159.94939),new n.zr(101,160.95433)]),F(k(l.I7,1),qt,3,0,[new n.zr(67,127.94826),new n.zr(68,128.94316),new n.zr(69,129.94045),new n.zr(70,130.9358),new n.zr(71,131.93375),new n.zr(72,132.92972),new n.zr(73,133.92849),new n.zr(74,134.924617),new n.zr(75,135.923447865),new n.zr(76,136.920713),new n.zr(77,137.920432261),new n.zr(78,138.916759814),new n.zr(79,139.915801649),new n.zr(80,140.913606636),new n.zr(81,141.912950738),new n.zr(82,142.910927571),new n.zr(83,143.912585768),new n.zr(84,144.912743879),new n.zr(85,145.914692165),new n.zr(86,146.915133898),new n.zr(87,147.917467786),new n.zr(88,148.918329195),new n.zr(89,149.920979477),new n.zr(90,150.921202693),new n.zr(91,151.923490557),new n.zr(92,152.924113189),new n.zr(93,153.926547019),new n.zr(94,154.928097047),new n.zr(95,155.931060357),new n.zr(96,156.9332),new n.zr(97,157.93669),new n.zr(98,158.93913),new n.zr(99,159.94299),new n.zr(100,160.94586),new n.zr(101,161.95029),new n.zr(102,162.95352)]),F(k(l.I7,1),qt,3,0,[new n.zr(68,129.94863),new n.zr(69,130.94589),new n.zr(70,131.94082),new n.zr(71,132.93873),new n.zr(72,133.93402),new n.zr(73,134.93235),new n.zr(74,135.9283),new n.zr(75,136.927046709),new n.zr(76,137.92354),new n.zr(77,138.922302),new n.zr(78,139.918991),new n.zr(79,140.918468512),new n.zr(80,141.915193274),new n.zr(81,142.914623555),new n.zr(82,143.91199473),new n.zr(83,144.913405611),new n.zr(84,145.91303676),new n.zr(85,146.914893275),new n.zr(86,147.914817914),new n.zr(87,148.917179521),new n.zr(88,149.917271454),new n.zr(89,150.919928351),new n.zr(90,151.919728244),new n.zr(91,152.922093907),new n.zr(92,153.922205303),new n.zr(93,154.92463594),new n.zr(94,155.925526236),new n.zr(95,156.928354506),new n.zr(96,157.929987938),new n.zr(97,158.9332),new n.zr(98,159.93514),new n.zr(99,160.93883),new n.zr(100,161.94122),new n.zr(101,162.94536),new n.zr(102,163.94828),new n.zr(103,164.95298)]),F(k(l.I7,1),qt,3,0,[new n.zr(69,131.95416),new n.zr(70,132.9489),new n.zr(71,133.94632),new n.zr(72,134.94172),new n.zr(73,135.9395),new n.zr(74,136.93521),new n.zr(75,137.93345),new n.zr(76,138.92882915),new n.zr(77,139.928083921),new n.zr(78,140.924885867),new n.zr(79,141.923400033),new n.zr(80,142.920286634),new n.zr(81,143.918774116),new n.zr(82,144.916261285),new n.zr(83,145.917199714),new n.zr(84,146.916741206),new n.zr(85,147.918153775),new n.zr(86,148.917925922),new n.zr(87,149.919698294),new n.zr(88,150.919846022),new n.zr(89,151.921740399),new n.zr(90,152.921226219),new n.zr(91,153.922975386),new n.zr(92,154.922889429),new n.zr(93,155.924750855),new n.zr(94,156.925419435),new n.zr(95,157.927841923),new n.zr(96,158.9290845),new n.zr(97,159.931460406),new n.zr(98,160.93368),new n.zr(99,161.93704),new n.zr(100,162.93921),new n.zr(101,163.94299),new n.zr(102,164.94572),new n.zr(103,165.94997),new n.zr(104,166.95305)]),F(k(l.I7,1),qt,3,0,[new n.zr(72,135.94707),new n.zr(73,136.94465),new n.zr(74,137.93997),new n.zr(75,138.93808),new n.zr(76,139.933236934),new n.zr(77,140.93221),new n.zr(78,141.927908919),new n.zr(79,142.926738636),new n.zr(80,143.923390357),new n.zr(81,144.921687498),new n.zr(82,145.918305344),new n.zr(83,146.919089446),new n.zr(84,147.918109771),new n.zr(85,148.919336427),new n.zr(86,149.918655455),new n.zr(87,150.920344273),new n.zr(88,151.919787882),new n.zr(89,152.921746283),new n.zr(90,153.920862271),new n.zr(91,154.922618801),new n.zr(92,155.922119552),new n.zr(93,156.923956686),new n.zr(94,157.924100533),new n.zr(95,158.926385075),new n.zr(96,159.927050616),new n.zr(97,160.929665688),new n.zr(98,161.930981211),new n.zr(99,162.93399),new n.zr(100,163.93586),new n.zr(101,164.93938),new n.zr(102,165.9416),new n.zr(103,166.94557),new n.zr(104,167.94836),new n.zr(105,168.95287)]),F(k(l.I7,1),qt,3,0,[new n.zr(73,137.95287),new n.zr(74,138.94803),new n.zr(75,139.945367985),new n.zr(76,140.94116),new n.zr(77,141.939073781),new n.zr(78,142.93475),new n.zr(79,143.93253),new n.zr(80,144.92888),new n.zr(81,145.927180629),new n.zr(82,146.924037176),new n.zr(83,147.924298636),new n.zr(84,148.92324163),new n.zr(85,149.923654158),new n.zr(86,150.923098169),new n.zr(87,151.924071324),new n.zr(88,152.923430858),new n.zr(89,153.924686236),new n.zr(90,154.923500411),new n.zr(91,155.924743749),new n.zr(92,156.924021155),new n.zr(93,157.92541026),new n.zr(94,158.925343135),new n.zr(95,159.927164021),new n.zr(96,160.927566289),new n.zr(97,161.929484803),new n.zr(98,162.930643942),new n.zr(99,163.933347253),new n.zr(100,164.93488),new n.zr(101,165.93805),new n.zr(102,166.94005),new n.zr(103,167.94364),new n.zr(104,168.94622),new n.zr(105,169.95025),new n.zr(106,170.9533)]),F(k(l.I7,1),qt,3,0,[new n.zr(74,139.95379),new n.zr(75,140.95119),new n.zr(76,141.946695946),new n.zr(77,142.94383),new n.zr(78,143.93907),new n.zr(79,144.936717),new n.zr(80,145.932720118),new n.zr(81,146.930878496),new n.zr(82,147.927177882),new n.zr(83,148.927333981),new n.zr(84,149.925579728),new n.zr(85,150.92617963),new n.zr(86,151.924713874),new n.zr(87,152.925760865),new n.zr(88,153.924422046),new n.zr(89,154.92574895),new n.zr(90,155.924278273),new n.zr(91,156.925461256),new n.zr(92,157.924404637),new n.zr(93,158.92573566),new n.zr(94,159.925193718),new n.zr(95,160.926929595),new n.zr(96,161.926794731),new n.zr(97,162.928727532),new n.zr(98,163.929171165),new n.zr(99,164.931699828),new n.zr(100,165.932803241),new n.zr(101,166.935649025),new n.zr(102,167.93723),new n.zr(103,168.940303648),new n.zr(104,169.94267),new n.zr(105,170.94648),new n.zr(106,171.94911),new n.zr(107,172.95344)]),F(k(l.I7,1),qt,3,0,[new n.zr(75,141.95986),new n.zr(76,142.95469),new n.zr(77,143.95164),new n.zr(78,144.94688),new n.zr(79,145.9441),new n.zr(80,146.93984),new n.zr(81,147.937269),new n.zr(82,148.933789944),new n.zr(83,149.932760914),new n.zr(84,150.931680791),new n.zr(85,151.931740598),new n.zr(86,152.930194506),new n.zr(87,153.930596268),new n.zr(88,154.929079084),new n.zr(89,155.929001869),new n.zr(90,156.928188059),new n.zr(91,157.92894573),new n.zr(92,158.927708537),new n.zr(93,159.928725679),new n.zr(94,160.927851662),new n.zr(95,161.92909242),new n.zr(96,162.928730286),new n.zr(97,163.930230577),new n.zr(98,164.930319169),new n.zr(99,165.932281267),new n.zr(100,166.933126195),new n.zr(101,167.935496424),new n.zr(102,168.936868306),new n.zr(103,169.939614951),new n.zr(104,170.941461227),new n.zr(105,171.94482),new n.zr(106,172.94729),new n.zr(107,173.95115),new n.zr(108,174.95405)]),F(k(l.I7,1),qt,3,0,[new n.zr(76,143.96059),new n.zr(77,144.95746),new n.zr(78,145.95212),new n.zr(79,146.94931),new n.zr(80,147.94444),new n.zr(81,148.942780527),new n.zr(82,149.937171034),new n.zr(83,150.93746),new n.zr(84,151.935078452),new n.zr(85,152.935093125),new n.zr(86,153.932777294),new n.zr(87,154.933204273),new n.zr(88,155.931015001),new n.zr(89,156.931945517),new n.zr(90,157.929912),new n.zr(91,158.930680718),new n.zr(92,159.929078924),new n.zr(93,160.930001348),new n.zr(94,161.928774923),new n.zr(95,162.930029273),new n.zr(96,163.929196996),new n.zr(97,164.9307228),new n.zr(98,165.93028997),new n.zr(99,166.932045448),new n.zr(100,167.932367781),new n.zr(101,168.934588082),new n.zr(102,169.935460334),new n.zr(103,170.938025885),new n.zr(104,171.939352149),new n.zr(105,172.9424),new n.zr(106,173.94434),new n.zr(107,174.94793),new n.zr(108,175.95029),new n.zr(109,176.95437)]),F(k(l.I7,1),qt,3,0,[new n.zr(77,145.966495),new n.zr(78,146.961081),new n.zr(79,147.95755),new n.zr(80,148.95265),new n.zr(81,149.94967),new n.zr(82,150.944842),new n.zr(83,151.9443),new n.zr(84,152.942027631),new n.zr(85,153.940832325),new n.zr(86,154.939191562),new n.zr(87,155.939006895),new n.zr(88,156.936756069),new n.zr(89,157.936996),new n.zr(90,158.934808966),new n.zr(91,159.935090772),new n.zr(92,160.933398042),new n.zr(93,161.933970147),new n.zr(94,162.932647648),new n.zr(95,163.933450972),new n.zr(96,164.932432463),new n.zr(97,165.933553133),new n.zr(98,166.932848844),new n.zr(99,167.934170375),new n.zr(100,168.934211117),new n.zr(101,169.935797877),new n.zr(102,170.936425817),new n.zr(103,171.938396118),new n.zr(104,172.939600336),new n.zr(105,173.942164618),new n.zr(106,174.943832897),new n.zr(107,175.946991412),new n.zr(108,176.94904),new n.zr(109,177.95264),new n.zr(110,178.95534)]),F(k(l.I7,1),qt,3,0,[new n.zr(78,147.96676),new n.zr(79,148.96348),new n.zr(80,149.95799),new n.zr(81,150.954657965),new n.zr(82,151.950167),new n.zr(83,152.94921),new n.zr(84,153.945651145),new n.zr(85,154.945792),new n.zr(86,155.942847109),new n.zr(87,156.94265865),new n.zr(88,157.939857897),new n.zr(89,158.940153735),new n.zr(90,159.93756),new n.zr(91,160.937357719),new n.zr(92,161.93575),new n.zr(93,162.936265492),new n.zr(94,163.93452),new n.zr(95,164.935397592),new n.zr(96,165.933879623),new n.zr(97,166.934946862),new n.zr(98,167.933894465),new n.zr(99,168.93518712),new n.zr(100,169.934758652),new n.zr(101,170.936322297),new n.zr(102,171.936377696),new n.zr(103,172.938206756),new n.zr(104,173.938858101),new n.zr(105,174.941272494),new n.zr(106,175.942568409),new n.zr(107,176.945257126),new n.zr(108,177.946643396),new n.zr(109,178.95017),new n.zr(110,179.95233),new n.zr(111,180.95615)]),F(k(l.I7,1),qt,3,0,[new n.zr(79,149.972668),new n.zr(80,150.967147),new n.zr(81,151.96361),new n.zr(82,152.95869),new n.zr(83,153.9571),new n.zr(84,154.953641324),new n.zr(85,155.952907),new n.zr(86,156.950101536),new n.zr(87,157.948577981),new n.zr(88,158.946615113),new n.zr(89,159.945383),new n.zr(90,160.943047504),new n.zr(91,161.943222),new n.zr(92,162.941203796),new n.zr(93,163.941215),new n.zr(94,164.939605886),new n.zr(95,165.939762646),new n.zr(96,166.938307056),new n.zr(97,167.938698576),new n.zr(98,168.937648757),new n.zr(99,169.93847219),new n.zr(100,170.937909903),new n.zr(101,171.939082239),new n.zr(102,172.938926901),new n.zr(103,173.940333522),new n.zr(104,174.940767904),new n.zr(105,175.942682399),new n.zr(106,176.943754987),new n.zr(107,177.945951366),new n.zr(108,178.947324216),new n.zr(109,179.949879968),new n.zr(110,180.95197),new n.zr(111,181.95521),new n.zr(112,182.95757),new n.zr(113,183.96117)]),F(k(l.I7,1),qt,3,0,[new n.zr(82,153.96425),new n.zr(83,154.96276),new n.zr(84,155.959247),new n.zr(85,156.958127),new n.zr(86,157.95405528),new n.zr(87,158.954003),new n.zr(88,159.950713588),new n.zr(89,160.950330852),new n.zr(90,161.947202977),new n.zr(91,162.947057),new n.zr(92,163.944422),new n.zr(93,164.94454),new n.zr(94,165.94225),new n.zr(95,166.9426),new n.zr(96,167.94063),new n.zr(97,168.941158567),new n.zr(98,169.93965),new n.zr(99,170.94049),new n.zr(100,171.93945798),new n.zr(101,172.94065),new n.zr(102,173.940040159),new n.zr(103,174.941502991),new n.zr(104,175.941401828),new n.zr(105,176.943220013),new n.zr(106,177.943697732),new n.zr(107,178.945815073),new n.zr(108,179.94654876),new n.zr(109,180.949099124),new n.zr(110,181.950552893),new n.zr(111,182.953531012),new n.zr(112,183.95544788),new n.zr(113,184.95878),new n.zr(114,185.96092)]),F(k(l.I7,1),qt,3,0,[new n.zr(83,155.971689),new n.zr(84,156.968145),new n.zr(85,157.966368),new n.zr(86,158.96232309),new n.zr(87,159.961358),new n.zr(88,160.958372992),new n.zr(89,161.956556553),new n.zr(90,162.95431665),new n.zr(91,163.95357),new n.zr(92,164.950817),new n.zr(93,165.95047),new n.zr(94,166.948639),new n.zr(95,167.947787),new n.zr(96,168.94592),new n.zr(97,169.94609),new n.zr(98,170.94446),new n.zr(99,171.944739818),new n.zr(100,172.94459),new n.zr(101,173.944167937),new n.zr(102,174.94365),new n.zr(103,175.944740551),new n.zr(104,176.944471766),new n.zr(105,177.945750349),new n.zr(106,178.945934113),new n.zr(107,179.947465655),new n.zr(108,180.947996346),new n.zr(109,181.950152414),new n.zr(110,182.951373188),new n.zr(111,183.954009331),new n.zr(112,184.955559086),new n.zr(113,185.9585501),new n.zr(114,186.96041),new n.zr(115,187.96371)]),F(k(l.I7,1),qt,3,0,[new n.zr(84,157.973939),new n.zr(85,158.97228),new n.zr(86,159.968369),new n.zr(87,160.967089),new n.zr(88,161.962750303),new n.zr(89,162.962532),new n.zr(90,163.95898381),new n.zr(91,164.958335962),new n.zr(92,165.955019896),new n.zr(93,166.954672),new n.zr(94,167.951863),new n.zr(95,168.951759),new n.zr(96,169.948473988),new n.zr(97,170.94946),new n.zr(98,171.948228837),new n.zr(99,172.948884),new n.zr(100,173.94616),new n.zr(101,174.94677),new n.zr(102,175.94559),new n.zr(103,176.94662),new n.zr(104,177.945848364),new n.zr(105,178.947071733),new n.zr(106,179.946705734),new n.zr(107,180.948198054),new n.zr(108,181.948205519),new n.zr(109,182.950224458),new n.zr(110,183.950932553),new n.zr(111,184.953420586),new n.zr(112,185.954362204),new n.zr(113,186.957158365),new n.zr(114,187.958486954),new n.zr(115,188.96191222),new n.zr(116,189.963179541)]),F(k(l.I7,1),qt,3,0,[new n.zr(85,159.981485),new n.zr(86,160.977661),new n.zr(87,161.975707),new n.zr(88,162.971375872),new n.zr(89,163.970319),new n.zr(90,164.967050268),new n.zr(91,165.965211372),new n.zr(92,166.962564),new n.zr(93,167.961609),new n.zr(94,168.95883),new n.zr(95,169.958163),new n.zr(96,170.955547),new n.zr(97,171.955285),new n.zr(98,172.953062),new n.zr(99,173.952114),new n.zr(100,174.951393),new n.zr(101,175.95157),new n.zr(102,176.95027),new n.zr(103,177.950851081),new n.zr(104,178.949981038),new n.zr(105,179.95078768),new n.zr(106,180.950064596),new n.zr(107,181.951211444),new n.zr(108,182.950821349),new n.zr(109,183.952524289),new n.zr(110,184.952955747),new n.zr(111,185.954986529),new n.zr(112,186.955750787),new n.zr(113,187.958112287),new n.zr(114,188.959228359),new n.zr(115,189.961816139),new n.zr(116,190.963123592),new n.zr(117,191.96596)]),F(k(l.I7,1),qt,3,0,[new n.zr(86,161.983819),new n.zr(87,162.982048),new n.zr(88,163.977927),new n.zr(89,164.976475),new n.zr(90,165.971934911),new n.zr(91,166.971554),new n.zr(92,167.967832911),new n.zr(93,168.967076205),new n.zr(94,169.963569716),new n.zr(95,170.96304),new n.zr(96,171.960078),new n.zr(97,172.959791),new n.zr(98,173.956307704),new n.zr(99,174.95708),new n.zr(100,175.953757941),new n.zr(101,176.955045),new n.zr(102,177.953348225),new n.zr(103,178.953951),new n.zr(104,179.952308241),new n.zr(105,180.953274494),new n.zr(106,181.952186222),new n.zr(107,182.95311),new n.zr(108,183.952490808),new n.zr(109,184.954043023),new n.zr(110,185.953838355),new n.zr(111,186.955747928),new n.zr(112,187.955835993),new n.zr(113,188.958144866),new n.zr(114,189.95844521),new n.zr(115,190.960927951),new n.zr(116,191.961479047),new n.zr(117,192.964148083),new n.zr(118,193.965179314),new n.zr(119,194.968123889),new n.zr(120,195.96962255)]),F(k(l.I7,1),qt,3,0,[new n.zr(88,164.98758),new n.zr(89,165.985506),new n.zr(90,166.980951577),new n.zr(91,167.979966),new n.zr(92,168.976390868),new n.zr(93,169.974441697),new n.zr(94,170.971779),new n.zr(95,171.970643),new n.zr(96,172.967707),new n.zr(97,173.966804),new n.zr(98,174.964279),new n.zr(99,175.963511),new n.zr(100,176.96117),new n.zr(101,177.960084944),new n.zr(102,178.95915),new n.zr(103,179.958555615),new n.zr(104,180.957642156),new n.zr(105,181.958127689),new n.zr(106,182.956814),new n.zr(107,183.957388318),new n.zr(108,184.95659),new n.zr(109,185.957951104),new n.zr(110,186.95736083),new n.zr(111,187.958851962),new n.zr(112,188.958716473),new n.zr(113,189.960592299),new n.zr(114,190.960591191),new n.zr(115,191.962602198),new n.zr(116,192.9629237),new n.zr(117,193.96507561),new n.zr(118,194.9659768),new n.zr(119,195.968379906),new n.zr(120,196.969636496),new n.zr(121,197.97228),new n.zr(122,198.973787159)]),F(k(l.I7,1),qt,3,0,[new n.zr(90,167.988035),new n.zr(91,168.986421),new n.zr(92,169.981734918),new n.zr(93,170.981251),new n.zr(94,171.977376138),new n.zr(95,172.976499642),new n.zr(96,173.972811276),new n.zr(97,174.972276),new n.zr(98,175.969),new n.zr(99,176.968453),new n.zr(100,177.964894223),new n.zr(101,178.965475),new n.zr(102,179.962023729),new n.zr(103,180.963177),new n.zr(104,181.961267637),new n.zr(105,182.961729),new n.zr(106,183.959851685),new n.zr(107,184.960753782),new n.zr(108,185.959432346),new n.zr(109,186.960697),new n.zr(110,187.959395697),new n.zr(111,188.9608319),new n.zr(112,189.959930073),new n.zr(113,190.961684653),new n.zr(114,191.961035158),new n.zr(115,192.962984504),new n.zr(116,193.962663581),new n.zr(117,194.964774449),new n.zr(118,195.964934884),new n.zr(119,196.967323401),new n.zr(120,197.967876009),new n.zr(121,198.970576213),new n.zr(122,199.971423885),new n.zr(123,200.974496467),new n.zr(124,201.97574)]),F(k(l.I7,1),qt,3,0,[new n.zr(92,170.991183),new n.zr(93,171.990109),new n.zr(94,172.986398138),new n.zr(95,173.984325861),new n.zr(96,174.981552),new n.zr(97,175.980269),new n.zr(98,176.977215),new n.zr(99,177.975975),new n.zr(100,178.973412),new n.zr(101,179.972396),new n.zr(102,180.969948),new n.zr(103,181.968621416),new n.zr(104,182.96762),new n.zr(105,183.966776046),new n.zr(106,184.965806956),new n.zr(107,185.965997671),new n.zr(108,186.964562),new n.zr(109,187.965321662),new n.zr(110,188.9642243),new n.zr(111,189.964698757),new n.zr(112,190.963649239),new n.zr(113,191.964810107),new n.zr(114,192.964131745),new n.zr(115,193.96533889),new n.zr(116,194.965017928),new n.zr(117,195.966551315),new n.zr(118,196.966551609),new n.zr(119,197.968225244),new n.zr(120,198.968748016),new n.zr(121,199.970717886),new n.zr(122,200.971640839),new n.zr(123,201.973788431),new n.zr(124,202.975137256),new n.zr(125,203.977705),new n.zr(126,204.97961)]),F(k(l.I7,1),qt,3,0,[new n.zr(95,174.991411),new n.zr(96,175.987413248),new n.zr(97,176.986336874),new n.zr(98,177.982476325),new n.zr(99,178.981783),new n.zr(100,179.978322),new n.zr(101,180.977806),new n.zr(102,181.97393546),new n.zr(103,182.974561),new n.zr(104,183.970705219),new n.zr(105,184.971983),new n.zr(106,185.969460021),new n.zr(107,186.969785),new n.zr(108,187.967511693),new n.zr(109,188.968733187),new n.zr(110,189.966958568),new n.zr(111,190.96706311),new n.zr(112,191.965921572),new n.zr(113,192.966644169),new n.zr(114,193.965381832),new n.zr(115,194.966638981),new n.zr(116,195.965814846),new n.zr(117,196.967195333),new n.zr(118,197.96675183),new n.zr(119,198.968262489),new n.zr(120,199.968308726),new n.zr(121,200.970285275),new n.zr(122,201.970625604),new n.zr(123,202.972857096),new n.zr(124,203.97347564),new n.zr(125,204.976056104),new n.zr(126,205.977498672),new n.zr(127,206.982577025),new n.zr(128,207.98594)]),F(k(l.I7,1),qt,3,0,[new n.zr(96,176.996881),new n.zr(97,177.994637),new n.zr(98,178.991466),new n.zr(99,179.990194),new n.zr(100,180.986904),new n.zr(101,181.98561),new n.zr(102,182.982697),new n.zr(103,183.98176),new n.zr(104,184.9791),new n.zr(105,185.977549881),new n.zr(106,186.97617),new n.zr(107,187.97592),new n.zr(108,188.974290451),new n.zr(109,189.974473379),new n.zr(110,190.972261952),new n.zr(111,191.972770785),new n.zr(112,192.970548),new n.zr(113,193.971053),new n.zr(114,194.96965),new n.zr(115,195.970515),new n.zr(116,196.9695362),new n.zr(117,197.970466294),new n.zr(118,198.969813837),new n.zr(119,199.970945394),new n.zr(120,200.97080377),new n.zr(121,201.972090569),new n.zr(122,202.972329088),new n.zr(123,203.973848646),new n.zr(124,204.97441227),new n.zr(125,205.976095321),new n.zr(126,206.977407908),new n.zr(127,207.982004653),new n.zr(128,208.985349125),new n.zr(129,209.990065574)]),F(k(l.I7,1),qt,3,0,[new n.zr(99,180.996714),new n.zr(100,181.992676101),new n.zr(101,182.99193),new n.zr(102,183.988198),new n.zr(103,184.98758),new n.zr(104,185.983485388),new n.zr(105,186.98403),new n.zr(106,187.979869108),new n.zr(107,188.98088),new n.zr(108,189.978180008),new n.zr(109,190.9782),new n.zr(110,191.975719811),new n.zr(111,192.97608),new n.zr(112,193.974648056),new n.zr(113,194.975920279),new n.zr(114,195.97271),new n.zr(115,196.97338),new n.zr(116,197.97198),new n.zr(117,198.972909384),new n.zr(118,199.97181556),new n.zr(119,200.972846589),new n.zr(120,201.972143786),new n.zr(121,202.973375491),new n.zr(122,203.973028761),new n.zr(123,204.974467112),new n.zr(124,205.974449002),new n.zr(125,206.975880605),new n.zr(126,207.97663585),new n.zr(127,208.981074801),new n.zr(128,209.984173129),new n.zr(129,210.988731474),new n.zr(130,211.991887495),new n.zr(131,212.9965),new n.zr(132,213.999798147)]),F(k(l.I7,1),qt,3,0,[new n.zr(102,184.997708),new n.zr(103,185.99648),new n.zr(104,186.993458),new n.zr(105,187.992173),new n.zr(106,188.989505),new n.zr(107,189.987520007),new n.zr(108,190.986053),new n.zr(109,191.985368),new n.zr(110,192.983662229),new n.zr(111,193.983430186),new n.zr(112,194.98112697),new n.zr(113,195.981236107),new n.zr(114,196.978934287),new n.zr(115,197.979024396),new n.zr(116,198.977576953),new n.zr(117,199.978141983),new n.zr(118,200.976970721),new n.zr(119,201.977674504),new n.zr(120,202.976868118),new n.zr(121,203.977805161),new n.zr(122,204.977374688),new n.zr(123,205.978482854),new n.zr(124,206.978455217),new n.zr(125,207.979726699),new n.zr(126,208.980383241),new n.zr(127,209.984104944),new n.zr(128,210.987258139),new n.zr(129,211.991271542),new n.zr(130,212.994374836),new n.zr(131,213.998698664),new n.zr(132,215.001832349),new n.zr(133,216.006199)]),F(k(l.I7,1),qt,3,0,[new n.zr(106,189.994293888),new n.zr(107,190.994653),new n.zr(108,191.99033039),new n.zr(109,192.991102),new n.zr(110,193.988284107),new n.zr(111,194.988045),new n.zr(112,195.985469432),new n.zr(113,196.985567),new n.zr(114,197.984024384),new n.zr(115,198.985044507),new n.zr(116,199.981735),new n.zr(117,200.982209),new n.zr(118,201.980704),new n.zr(119,202.981412863),new n.zr(120,203.980307113),new n.zr(121,204.981165396),new n.zr(122,205.980465241),new n.zr(123,206.981578228),new n.zr(124,207.981231059),new n.zr(125,208.982415788),new n.zr(126,209.982857396),new n.zr(127,210.986636869),new n.zr(128,211.988851755),new n.zr(129,212.992842522),new n.zr(130,213.995185949),new n.zr(131,214.999414609),new n.zr(132,216.001905198),new n.zr(133,217.006253),new n.zr(134,218.008965773)]),F(k(l.I7,1),qt,3,0,[new n.zr(108,193.000188),new n.zr(109,193.997973),new n.zr(110,194.996554),new n.zr(111,195.995702),new n.zr(112,196.993891293),new n.zr(113,197.99343368),new n.zr(114,198.991008569),new n.zr(115,199.990920883),new n.zr(116,200.988486908),new n.zr(117,201.988448629),new n.zr(118,202.986847216),new n.zr(119,203.987261559),new n.zr(120,204.986036352),new n.zr(121,205.986599242),new n.zr(122,206.985775861),new n.zr(123,207.986582508),new n.zr(124,208.986158678),new n.zr(125,209.987131308),new n.zr(126,210.987480806),new n.zr(127,211.990734657),new n.zr(128,212.99292115),new n.zr(129,213.996356412),new n.zr(130,214.998641245),new n.zr(131,216.002408839),new n.zr(132,217.004709619),new n.zr(133,218.008681458),new n.zr(134,219.011296478),new n.zr(135,220.015301),new n.zr(136,221.01814),new n.zr(137,222.02233),new n.zr(138,223.02534)]),F(k(l.I7,1),qt,3,0,[new n.zr(110,196.001117268),new n.zr(111,197.001661),new n.zr(112,197.998779978),new n.zr(113,198.998309),new n.zr(114,199.995634148),new n.zr(115,200.995535),new n.zr(116,201.993899382),new n.zr(117,202.994765192),new n.zr(118,203.991365),new n.zr(119,204.991668),new n.zr(120,205.99016),new n.zr(121,206.990726826),new n.zr(122,207.989631237),new n.zr(123,208.990376634),new n.zr(124,209.989679862),new n.zr(125,210.99058541),new n.zr(126,211.990688899),new n.zr(127,212.993868354),new n.zr(128,213.995346275),new n.zr(129,214.998729195),new n.zr(130,216.000258153),new n.zr(131,217.003914555),new n.zr(132,218.005586315),new n.zr(133,219.009474831),new n.zr(134,220.011384149),new n.zr(135,221.015455),new n.zr(136,222.017570472),new n.zr(137,223.02179),new n.zr(138,224.02409),new n.zr(139,225.02844),new n.zr(140,226.03089),new n.zr(141,227.035407),new n.zr(142,228.038084)]),F(k(l.I7,1),qt,3,0,[new n.zr(113,200.006499),new n.zr(114,201.00458692),new n.zr(115,202.00396885),new n.zr(116,203.001423829),new n.zr(117,204.001221209),new n.zr(118,204.998663961),new n.zr(119,205.998486886),new n.zr(120,206.996859385),new n.zr(121,207.997133849),new n.zr(122,208.995915421),new n.zr(123,209.996398327),new n.zr(124,210.995529332),new n.zr(125,211.996194988),new n.zr(126,212.996174845),new n.zr(127,213.99895474),new n.zr(128,215.000326029),new n.zr(129,216.003187873),new n.zr(130,217.004616452),new n.zr(131,218.007563326),new n.zr(132,219.009240843),new n.zr(133,220.012312978),new n.zr(134,221.014245654),new n.zr(135,222.017543957),new n.zr(136,223.019730712),new n.zr(137,224.023235513),new n.zr(138,225.025606914),new n.zr(139,226.029343423),new n.zr(140,227.031833167),new n.zr(141,228.034776087),new n.zr(142,229.038426),new n.zr(143,230.04251),new n.zr(144,231.045407),new n.zr(145,232.049654)]),F(k(l.I7,1),qt,3,0,[new n.zr(115,203.00921),new n.zr(116,204.006434513),new n.zr(117,205.006187),new n.zr(118,206.004463814),new n.zr(119,207.005176607),new n.zr(120,208.001776),new n.zr(121,209.001944),new n.zr(122,210.000446),new n.zr(123,211.000893996),new n.zr(124,211.999783492),new n.zr(125,213.000345847),new n.zr(126,214.000091141),new n.zr(127,215.002704195),new n.zr(128,216.003518402),new n.zr(129,217.00630601),new n.zr(130,218.007123948),new n.zr(131,219.010068787),new n.zr(132,220.011014669),new n.zr(133,221.013907762),new n.zr(134,222.01536182),new n.zr(135,223.01849714),new n.zr(136,224.020202004),new n.zr(137,225.023604463),new n.zr(138,226.025402555),new n.zr(139,227.029170677),new n.zr(140,228.031064101),new n.zr(141,229.034820309),new n.zr(142,230.037084774),new n.zr(143,231.04122),new n.zr(144,232.043693),new n.zr(145,233.047995),new n.zr(146,234.050547)]),F(k(l.I7,1),qt,3,0,[new n.zr(118,207.012469754),new n.zr(119,208.012112949),new n.zr(120,209.009568736),new n.zr(121,210.009256802),new n.zr(122,211.007648196),new n.zr(123,212.007811441),new n.zr(124,213.006573689),new n.zr(125,214.006893072),new n.zr(126,215.006450832),new n.zr(127,216.008721268),new n.zr(128,217.009332676),new n.zr(129,218.011625045),new n.zr(130,219.012404918),new n.zr(131,220.014752105),new n.zr(132,221.015575746),new n.zr(133,222.017828852),new n.zr(134,223.01912603),new n.zr(135,224.021708435),new n.zr(136,225.023220576),new n.zr(137,226.026089848),new n.zr(138,227.027746979),new n.zr(139,228.031014825),new n.zr(140,229.032930871),new n.zr(141,230.036025144),new n.zr(142,231.038551503),new n.zr(143,232.042022474),new n.zr(144,233.04455),new n.zr(145,234.04842),new n.zr(146,235.051102),new n.zr(147,236.055178)]),F(k(l.I7,1),qt,3,0,[new n.zr(120,210.015711883),new n.zr(121,211.016306912),new n.zr(122,212.012916),new n.zr(123,213.012962),new n.zr(124,214.011451),new n.zr(125,215.011726597),new n.zr(126,216.011050963),new n.zr(127,217.013066169),new n.zr(128,218.013267744),new n.zr(129,219.015521253),new n.zr(130,220.015733126),new n.zr(131,221.018171499),new n.zr(132,222.018454131),new n.zr(133,223.020795153),new n.zr(134,224.02145925),new n.zr(135,225.023941441),new n.zr(136,226.024890681),new n.zr(137,227.027698859),new n.zr(138,228.028731348),new n.zr(139,229.03175534),new n.zr(140,230.033126574),new n.zr(141,231.03629706),new n.zr(142,232.03805036),new n.zr(143,233.041576923),new n.zr(144,234.043595497),new n.zr(145,235.04750442),new n.zr(146,236.04971),new n.zr(147,237.053894),new n.zr(148,238.056243)]),F(k(l.I7,1),qt,3,0,[new n.zr(122,213.021183209),new n.zr(123,214.02073923),new n.zr(124,215.019097612),new n.zr(125,216.019109649),new n.zr(126,217.018288571),new n.zr(127,218.020007906),new n.zr(128,219.019880348),new n.zr(129,220.021876493),new n.zr(130,221.021863742),new n.zr(131,222.023726),new n.zr(132,223.023963748),new n.zr(133,224.025614854),new n.zr(134,225.026115172),new n.zr(135,226.02793275),new n.zr(136,227.028793151),new n.zr(137,228.031036942),new n.zr(138,229.032088601),new n.zr(139,230.034532562),new n.zr(140,231.035878898),new n.zr(141,232.03858172),new n.zr(142,233.040240235),new n.zr(143,234.043302325),new n.zr(144,235.045436759),new n.zr(145,236.048675176),new n.zr(146,237.05113943),new n.zr(147,238.054497046),new n.zr(148,239.05713),new n.zr(149,240.06098)]),F(k(l.I7,1),qt,3,0,[new n.zr(126,218.023487),new n.zr(127,219.024915423),new n.zr(128,220.024712),new n.zr(129,221.026351),new n.zr(130,222.02607),new n.zr(131,223.027722956),new n.zr(132,224.027590139),new n.zr(133,225.029384369),new n.zr(134,226.02933975),new n.zr(135,227.031140069),new n.zr(136,228.031366357),new n.zr(137,229.033496137),new n.zr(138,230.033927392),new n.zr(139,231.036289158),new n.zr(140,232.03714628),new n.zr(141,233.039628196),new n.zr(142,234.040945606),new n.zr(143,235.043923062),new n.zr(144,236.045561897),new n.zr(145,237.048723955),new n.zr(146,238.050782583),new n.zr(147,239.054287777),new n.zr(148,240.056585734),new n.zr(149,241.06033),new n.zr(150,242.062925)]),F(k(l.I7,1),qt,3,0,[new n.zr(132,225.033899689),new n.zr(133,226.035129),new n.zr(134,227.034958261),new n.zr(135,228.03618),new n.zr(136,229.036246866),new n.zr(137,230.037812591),new n.zr(138,231.038233161),new n.zr(139,232.040099),new n.zr(140,233.04073235),new n.zr(141,234.042888556),new n.zr(142,235.044055876),new n.zr(143,236.046559724),new n.zr(144,237.048167253),new n.zr(145,238.050940464),new n.zr(146,239.052931399),new n.zr(147,240.056168828),new n.zr(148,241.058246266),new n.zr(149,242.061635),new n.zr(150,243.064273),new n.zr(151,244.06785)]),F(k(l.I7,1),qt,3,0,[new n.zr(134,228.038727686),new n.zr(135,229.040138934),new n.zr(136,230.039645603),new n.zr(137,231.041258),new n.zr(138,232.041179445),new n.zr(139,233.04298757),new n.zr(140,234.043304681),new n.zr(141,235.0452815),new n.zr(142,236.046048088),new n.zr(143,237.048403774),new n.zr(144,238.0495534),new n.zr(145,239.052156519),new n.zr(146,240.05380746),new n.zr(147,241.056845291),new n.zr(148,242.058736847),new n.zr(149,243.061997013),new n.zr(150,244.06419765),new n.zr(151,245.067738657),new n.zr(152,246.070198429),new n.zr(153,247.07407)]),F(k(l.I7,1),qt,3,0,[new n.zr(136,231.04556),new n.zr(137,232.04659),new n.zr(138,233.046472),new n.zr(139,234.047794),new n.zr(140,235.048029),new n.zr(141,236.049569),new n.zr(142,237.049970748),new n.zr(143,238.051977839),new n.zr(144,239.053018481),new n.zr(145,240.055287826),new n.zr(146,241.056822944),new n.zr(147,242.059543039),new n.zr(148,243.061372686),new n.zr(149,244.064279429),new n.zr(150,245.066445398),new n.zr(151,246.069768438),new n.zr(152,247.072086),new n.zr(153,248.075745),new n.zr(154,249.07848)]),F(k(l.I7,1),qt,3,0,[new n.zr(137,233.0508),new n.zr(138,234.05024),new n.zr(139,235.051591),new n.zr(140,236.051405),new n.zr(141,237.052891),new n.zr(142,238.053016298),new n.zr(143,239.054951),new n.zr(144,240.055519046),new n.zr(145,241.057646736),new n.zr(146,242.058829326),new n.zr(147,243.061382249),new n.zr(148,244.062746349),new n.zr(149,245.065485586),new n.zr(150,246.067217551),new n.zr(151,247.070346811),new n.zr(152,248.072342247),new n.zr(153,249.075947062),new n.zr(154,250.078350687),new n.zr(155,251.082277873),new n.zr(156,252.08487)]),F(k(l.I7,1),qt,3,0,[new n.zr(138,235.05658),new n.zr(139,236.05733),new n.zr(140,237.057127),new n.zr(141,238.058266),new n.zr(142,239.058362),new n.zr(143,240.059749),new n.zr(144,241.060223),new n.zr(145,242.06205),new n.zr(146,243.06300157),new n.zr(147,244.065167882),new n.zr(148,245.066355386),new n.zr(149,246.068666836),new n.zr(150,247.070298533),new n.zr(151,248.07308),new n.zr(152,249.074979937),new n.zr(153,250.078310529),new n.zr(154,251.08075344),new n.zr(155,252.084303),new n.zr(156,253.08688),new n.zr(157,254.0906)]),F(k(l.I7,1),qt,3,0,[new n.zr(139,237.06207),new n.zr(140,238.06141),new n.zr(141,239.062579),new n.zr(142,240.062295),new n.zr(143,241.063716),new n.zr(144,242.063688713),new n.zr(145,243.065421),new n.zr(146,244.06599039),new n.zr(147,245.068039),new n.zr(148,246.068798807),new n.zr(149,247.070992043),new n.zr(150,248.07217808),new n.zr(151,249.074846818),new n.zr(152,250.076399951),new n.zr(153,251.079580056),new n.zr(154,252.081619582),new n.zr(155,253.085126791),new n.zr(156,254.087316198),new n.zr(157,255.091039),new n.zr(158,256.09344)]),F(k(l.I7,1),qt,3,0,[new n.zr(141,240.06892),new n.zr(142,241.068662),new n.zr(143,242.069699),new n.zr(144,243.069631),new n.zr(145,244.070969),new n.zr(146,245.071317),new n.zr(147,246.072965),new n.zr(148,247.07365),new n.zr(149,248.075458),new n.zr(150,249.076405),new n.zr(151,250.078654),new n.zr(152,251.079983592),new n.zr(153,252.082972247),new n.zr(154,253.084817974),new n.zr(155,254.088016026),new n.zr(156,255.090266386),new n.zr(157,256.093592),new n.zr(158,257.095979)]),F(k(l.I7,1),qt,3,0,[new n.zr(142,242.07343),new n.zr(143,243.07451),new n.zr(144,244.074077),new n.zr(145,245.075375),new n.zr(146,246.075281634),new n.zr(147,247.076819),new n.zr(148,248.077184411),new n.zr(149,249.079024),new n.zr(150,250.079514759),new n.zr(151,251.081566467),new n.zr(152,252.082460071),new n.zr(153,253.085176259),new n.zr(154,254.086847795),new n.zr(155,255.089955466),new n.zr(156,256.091766522),new n.zr(157,257.095098635),new n.zr(158,258.097069),new n.zr(159,259.100588)]),F(k(l.I7,1),qt,3,0,[new n.zr(144,245.081017),new n.zr(145,246.081933),new n.zr(146,247.081804),new n.zr(147,248.082909),new n.zr(148,249.083002),new n.zr(149,250.084488),new n.zr(150,251.084919),new n.zr(151,252.08663),new n.zr(152,253.08728),new n.zr(153,254.089725),new n.zr(154,255.091075196),new n.zr(155,256.094052757),new n.zr(156,257.095534643),new n.zr(157,258.098425321),new n.zr(158,259.100503),new n.zr(159,260.103645)]),F(k(l.I7,1),qt,3,0,[new n.zr(147,249.087823),new n.zr(148,250.087493),new n.zr(149,251.08896),new n.zr(150,252.088965909),new n.zr(151,253.090649),new n.zr(152,254.090948746),new n.zr(153,255.093232449),new n.zr(154,256.094275879),new n.zr(155,257.096852778),new n.zr(156,258.0982),new n.zr(157,259.101024),new n.zr(158,260.102636),new n.zr(159,261.105743),new n.zr(160,262.10752)]),F(k(l.I7,1),qt,3,0,[new n.zr(148,251.09436),new n.zr(149,252.09533),new n.zr(150,253.095258),new n.zr(151,254.096587),new n.zr(152,255.096769),new n.zr(153,256.098763),new n.zr(154,257.099606),new n.zr(155,258.101883),new n.zr(156,259.10299),new n.zr(157,260.105572),new n.zr(158,261.106941),new n.zr(159,262.109692),new n.zr(160,263.111394)]),F(k(l.I7,1),qt,3,0,[new n.zr(149,253.100679),new n.zr(150,254.100166),new n.zr(151,255.101492),new n.zr(152,256.101179573),new n.zr(153,257.103072),new n.zr(154,258.103568),new n.zr(155,259.105628),new n.zr(156,260.106434),new n.zr(157,261.108752),new n.zr(158,262.109918),new n.zr(159,263.11254),new n.zr(160,264.113978)]),F(k(l.I7,1),qt,3,0,[new n.zr(150,255.107398),new n.zr(151,256.10811),new n.zr(152,257.107858),new n.zr(153,258.109438),new n.zr(154,259.109721),new n.zr(155,260.111427),new n.zr(156,261.112106),new n.zr(157,262.114153),new n.zr(158,263.115078),new n.zr(159,264.117473),new n.zr(160,265.118659)]),F(k(l.I7,1),qt,3,0,[new n.zr(152,258.113151),new n.zr(153,259.114652),new n.zr(154,260.114435447),new n.zr(155,261.116199),new n.zr(156,262.116477),new n.zr(157,263.118313),new n.zr(158,264.118924),new n.zr(159,265.121066),new n.zr(160,266.121928)]),F(k(l.I7,1),qt,3,0,[new n.zr(153,260.121803),new n.zr(154,261.1218),new n.zr(155,262.123009),new n.zr(156,263.123146),new n.zr(157,264.12473),new n.zr(158,265.125198),new n.zr(159,266.127009),new n.zr(160,267.12774)]),F(k(l.I7,1),qt,3,0,[new n.zr(155,263.12871),new n.zr(156,264.128408258),new n.zr(157,265.130001),new n.zr(158,266.130042),new n.zr(159,267.131774),new n.zr(160,268.132156),new n.zr(161,269.134114)]),F(k(l.I7,1),qt,3,0,[new n.zr(156,265.136567),new n.zr(157,266.13794),new n.zr(158,267.137526),new n.zr(159,268.138816),new n.zr(160,269.139106),new n.zr(161,270.140723),new n.zr(162,271.141229)])]);},n.Cr=function(t,e){n.Br();var A,P;for(P=e-t,A=0;A<n.Ar[t].length;A++)if(n.Ar[t][A].b==P)return n.Ar[t][A].a;return NaN;},n.Dr=function(t,e){var A;if(t.b.length!=e.b.length)return t.b.length<e.b.length?-1:1;for(A=0;A<t.b.length;A++)if(t.b[A]!=e.b[A])return t.b[A]<e.b[A]?-1:1;return 0;},n.Er=function(t){var e,A,P,r,i,D;for(A=0,r=t.a,i=0,D=r.length;i<D;++i)P=r[i],P&&++A;for(t.b=z(l.d7,H,6,A,15,1),A=0,e=0;e<t.a.length;e++)t.a[e]&&(t.b[A++]=e);},n.Fr=function(t,e){var A;for(n.Er(t),A=0;A<t.b.length;A++)if(e[t.b[A]])return  true;return  false;},n.Gr=function(t){this.a=z(l.zeb,ht,6,t,16,1);},x(239,1,{239:1,32:1},n.Gr),y.Rb=function(t){return n.Dr(this,t);},l.J7=X(239),n.Lr=function(){n.Lr=o1,n.Kr=F(k(l.b7,1),jt,6,15,[0,1.00794,4.0026,6.941,9.0122,10.811,12.011,14.007,15.999,18.998,20.18,22.99,24.305,26.982,28.086,30.974,32.066,35.453,39.948,39.098,40.078,44.956,47.867,50.942,51.996,54.938,55.845,58.933,58.693,63.546,65.39,69.723,72.61,74.922,78.96,79.904,83.8,85.468,87.62,88.906,91.224,92.906,95.94,98.906,101.07,102.91,106.42,107.87,112.41,114.82,118.71,121.76,127.6,126.9,131.29,132.91,137.33,138.91,140.12,140.91,144.24,146.92,150.36,151.96,157.25,158.93,162.5,164.93,167.26,168.93,173.04,174.97,178.49,180.95,183.84,186.21,190.23,192.22,195.08,196.97,200.59,204.38,207.2,208.98,209.98,209.99,222.02,223.02,226.03,227.03,232.04,231.04,238.03,237.05,239.05,241.06,244.06,249.08,252.08,252.08,257.1,258.1,259.1,262.11,267.12,268.13,271.13,270.13,277.15,276.15,nP,nP,283.17,285.18,AP,AP,293.2,PP,PP,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.0141,3.016,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71.0787,156.18828,114.10364,115.0877,103.1447,128.13052,129.11458,57.05182,137.14158,113.15934,113.15934,128.17428,131.19846,147.17646,97.11658,87.0777,101.10458,186.2134,163.17546,99.13246]),n.Ir=F(k(l.b7,1),jt,6,15,[0,1.007825,4.0026,7.016003,9.012182,11.009305,12,14.003074,15.994915,18.998403,19.992435,22.989767,23.985042,26.98153,27.976927,30.973762,31.97207,34.968852,39.962384,38.963707,39.962591,44.95591,47.947947,50.943962,51.940509,54.938047,55.934939,58.933198,57.935346,62.939598,63.929145,68.92558,73.921177,74.921594,79.91652,78.918336,83.911507,84.911794,87.905619,88.905849,89.904703,92.906377,97.905406,89.92381,101.904348,102.9055,105.903478,106.905092,113.903357,114.90388,119.9022,120.903821,129.906229,126.904473,131.904144,132.905429,137.905232,138.906346,139.905433,140.907647,141.907719,135.92398,151.919729,152.921225,157.924099,158.925342,163.929171,164.930319,165.93029,168.934212,173.938859,174.94077,179.946545,180.947992,183.950928,186.955744,191.961467,192.962917,194.964766,196.966543,201.970617,204.974401,207.976627,208.980374,193.98818,195.99573,199.9957,201.00411,206.0038,210.00923,232.038054,216.01896,238.050784,229.03623,232.041169,237.05005,238.05302,242.06194,240.06228,243.06947,243.07446,248.08275,251.08887,253.09515,257.10295,257.10777,271.13,270.13,277.15,276.15,nP,nP,283.17,285.18,AP,AP,291.2,PP,PP,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.014,3.01605,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]),n.Jr=F(k(l.d7,1),H,6,15,[6,1,7,8]);},n.Mr=function(t){var e,A;for(A=t.a,e=0;e<t.b.length;e++)A+=t.b[e]*n.Ir[t.c[e]];return A;},n.Nr=function(t){var e,A;for(e=new T.Gjb(),A=0;A<t.b.length;A++)T.Fjb(e,(n.Qk(),n.Kk)[t.c[A]]),t.b[A]>1&&T.Ejb(e,t.b[A]);return e.a;},n.Or=function(t){var e,A;for(A=t.d,e=0;e<t.b.length;e++)A+=t.b[e]*n.Kr[t.c[e]];return A;},n.Pr=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(n.aw(t,1),r=z(l.d7,H,6,191,15,1),A=0;A<t.q;A++)switch(t.C[A]){case 171:r[1]+=5,r[6]+=3,r[7]+=1,r[8]+=1;break;case 172:r[1]+=12,r[6]+=6,r[7]+=4,r[8]+=1;break;case 173:r[1]+=6,r[6]+=4,r[7]+=2,r[8]+=2;break;case 174:r[1]+=5,r[6]+=4,r[7]+=1,r[8]+=3;break;case 175:r[1]+=5,r[6]+=3,r[7]+=1,r[8]+=1,r[16]+=1;break;case 176:r[1]+=8,r[6]+=5,r[7]+=2,r[8]+=2;break;case 177:r[1]+=7,r[6]+=5,r[7]+=1,r[8]+=3;break;case 178:r[1]+=3,r[6]+=2,r[7]+=1,r[8]+=1;break;case 179:r[1]+=7,r[6]+=6,r[7]+=3,r[8]+=1;break;case 181:case 180:r[1]+=11,r[6]+=6,r[7]+=1,r[8]+=1;break;case 182:r[1]+=12,r[6]+=6,r[7]+=2,r[8]+=1;break;case 183:r[1]+=9,r[6]+=5,r[7]+=1,r[8]+=1,r[16]+=1;break;case 184:r[1]+=9,r[6]+=9,r[7]+=1,r[8]+=1;break;case 185:r[1]+=7,r[6]+=5,r[7]+=1,r[8]+=1;break;case 186:r[1]+=5,r[6]+=3,r[7]+=1,r[8]+=2;break;case 187:r[1]+=7,r[6]+=4,r[7]+=1,r[8]+=2;break;case 188:r[1]+=10,r[6]+=11,r[7]+=2,r[8]+=1;break;case 189:r[1]+=9,r[6]+=9,r[7]+=1,r[8]+=2;break;case 190:r[1]+=9,r[6]+=5,r[7]+=1,r[8]+=1;break;case 1:switch(t.A[A]){case 0:case 1:++r[1];break;case 2:++r[151];break;case 3:++r[152];}break;default:++r[t.C[A]];}for(P=0;P<t.q;P++)t.C[P]>=171&&t.C[P]<=190?r[1]+=2-n.up(t,P):r[1]+=n.np(t,P);for(u=0,Q=1;Q<=190;Q++)r[Q]!=0&&++u;for(this.b=z(l.d7,H,6,u,15,1),this.c=z(l.d7,H,6,u,15,1),u=0,o=0;o<n.Jr.length;o++)r[n.Jr[o]]!=0&&(this.b[u]=r[n.Jr[o]],this.c[u]=n.Jr[o],++u,r[n.Jr[o]]=0);for(;;){for(s="zzz",L=-1,D=1;D<=190;D++)r[D]>0&&T.$ib(s,(n.Qk(),n.Kk)[D])>0&&(s=(n.Qk(),n.Kk)[D],L=D);if(L==-1)break;this.b[u]=r[L],this.c[u]=L,++u,r[L]=0;}for(this.a=0,this.d=0,e=0;e<t.f;e++)t.C[e]!=1&&t.A[e]!=0&&(D=t.C[e],i=t.A[e],this.a+=n.Cr(D,i)-n.Ir[D],this.d+=n.Cr(D,i)-n.Kr[D]);},x(152,1,{152:1}),y.lb=function(t){var e;if(t===this)return  true;if(!Jt(t,152))return  false;for(e=0;e<this.b.length;e++)if(this.b[e]!=t.b[e])return  false;return  true;},y.a=0,y.d=0,l.K7=X(152),n.Rr=function(t){switch(t){case 6:return 1;case 53:return 2;case 33:return 3;case 34:return 4;case 35:return 5;case 15:return 6;case 16:return 7;case 17:return 8;case 7:return 9;case 8:return 10;case 9:return 11;}return 0;},n.Sr=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N;for(n.aw(t,7),K=0,b=z(l.zeb,ht,6,t.f,16,1),N=z(l.d7,H,6,t.f,15,1),i=0;i<t.g;i++)A=t.D[0][i],P=t.D[1][i],D=t.s[A],u=t.s[P],D!=0&&u!=0&&D<0^u<0&&(b[A]=true,b[P]=true),N[A]+=n.em(t,i),N[P]+=n.em(t,i);for(o=0;o<t.f;o++)if(K+=t.s[o],t.s[o]==1){if(t.C[o]==7&&!b[o]){if(N[o]<=3)K-=1,t.s[o]=0,t.T=0,t.j[o]!=t.e[o]&&(n.pl(t,n.Yo(t,o,t.e[o]-1)),n.aw(t,7));else if(o<t.f&&n.nt(t.p,o)){for(m=z(l.zeb,ht,6,t.f,16,1),C=z(l.zeb,ht,6,t.g,16,1),n.Ho(t,o,true,m,C),L=0;L<t.f;L++)if(m[L]&&t.C[L]==7&&t.s[L]==0&&N[L]==2&&n.Vr(t,C,o,L)){K-=1;break;}}}}else if(t.s[o]<0&&(t.C[o]==6||t.C[o]==7||t.C[o]==8||t.C[o]==16))if(b[o]){for(S=z(l.d7,H,6,16,15,1),S[0]=o,O=1,V=0;V<O;){for(r=0;r<t.r;r++)if(e=-1,t.D[0][r]==S[V]?e=t.D[1][r]:t.D[1][r]==S[V]&&(e=t.D[0][r]),e!=-1&&t.s[e]!=0){for(s=false,g=0;g<O;g++)if(e==S[g]){s=true;break;}s||(O==S.length&&(Q=z(l.d7,H,6,2*S.length,15,1),T.Vjb(S,0,Q,0,S.length),S=Q),S[O++]=e);}++V;}for(h=0,d=0;d<O;d++)h+=n.Al(t,S[d]);if(h<0){for(v=-1,M=99,c=0;c<O;c++)n.Al(t,S[c])<0&&M>n.Rr(n.Sl(t,S[c]))&&(M=n.Rr(n.Sl(t,S[c])),v=S[c]);v!=-1&&(K-=t.s[v],t.s[v]=0,t.T=0);}}else K-=t.s[o],t.s[o]=0,t.T=0;return K>0&&(K=n.Tr(t,K)),K;},n.Tr=function(t,e){var A,P,r,i;if(e>0&&(e=n.Wr(t,e,9)),e>0&&(e=n.Wr(t,e,17)),e>0&&(e=n.Wr(t,e,35)),e>0&&(e=n.Wr(t,e,53)),e>0){for(A=0;A<t.f;A++)if(t.s[A]>0){for(r=false,i=0;i<t.j[A];i++)if(P=t.i[A][i],t.s[P]==0&&n.Gm(t,P)&&n.np(t,P)>0){if(t.s[P]=-1,t.T=0,--e,e==0)return 0;r=true;break;}if(r)continue;}}return e>0&&(e=n.Ur(t,e,8,16)),e>0&&(e=n.Ur(t,e,8,15)),e>0&&(e=n.Ur(t,e,8,6)),e>0&&(e=n.Ur(t,e,7,16)),e;},n.Ur=function(t,e,A,P){var r,i,D,u,o,Q,L,s;for(r=0;r<t.f;r++)if(t.C[r]==A&&t.s[r]==0&&n.np(t,r)>0)for(u=false,Q=0;Q<t.j[r];Q++){if(t.n[r][Q]==1&&(i=t.i[r][Q],t.C[i]==P)){for(s=false,o=false,L=0;L<t.j[i];L++){if(D=t.i[i][L],t.s[D]<0){o=true;break;}D!=r&&t.C[D]==8&&t.n[i][L]==2&&(s=true);}if(!o&&s){if(t.s[r]=-1,t.T=0,--e,e==0)return 0;u=true;}}if(u)break;}return e;},n.Vr=function(t,e,A,P){var r,i,D,u,o,Q,L,s;for(s=z(l.d7,H,6,t.g,15,1),o=z(l.zeb,ht,6,t.g,16,1),D=0;D<t.g;D++)s[D]=t.H[D],e[D]&&(o[D]=true,t.H[D]=1,t.T=0);for(r=0;r<t.f;r++)if(r!=P&&t.o[r]==0&&o[t.k[P][0]])for(L=0;L<t.j[r];L++)u=t.k[r][L],t.H[u]=1,t.T=0,o[u]=false;for(Q=0;Q<t.j[A];Q++)u=t.k[A][Q],t.H[u]=1,t.T=0,o[u]=false;if(n.Fe(new n.Ue(t),o,false))return t.s[A]=0,t.T=0,t.j[P]!=t.e[P]&&n.pl(t,n.Yo(t,P,t.e[P]-1)),n.aw(t,7),true;for(i=0;i<t.g;i++)n.Ln(t,i,s[i]);return  false;},n.Wr=function(t,e,A){var P;for(P=0;P<t.f;P++)if(t.C[P]==A&&t.s[P]==0&&t.j[P]==0&&(t.s[P]=-1,t.T=0,--e,e==0))return 0;return e;},n.Xr=function(t,e){var A,P;for(P=o0.Imb(t.b,e),A=w.Xzb(P).length;A<10;A++)T.Jjb(t.a,32);T.Njb(t.a,P);},n.Yr=function(t,e){var A,P,r;if(e<0||e>999){T.Njb(t.a,"  ?");return;}for(A=false,P=0;P<3;P++)r=e/100|0,r==0?P==2||A?T.Jjb(t.a,48):T.Jjb(t.a,32):(T.Jjb(t.a,48+r&r1),A=true),e=10*(e%100);},n.Zr=function(t,e){Rt.wgb(e,t.a.a);},n.$r=function(t){n._r.call(this,t);},n._r=function(t){n.bs.call(this,t,new T.Qjb());},n.as=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt,mt,y1,u1,Mt,Zt,m1,Et,f1;for(this.b=new o0.Nmb((wt=rP,wt)),Nt=(T.Ujb(),p1),n.aw(t,15),bt=true,r=0;r<t.f;r++)if((t.u[r]&3)!=0&&(t.u[r]&3)!=3&&(t.u[r]&U1)>>18!=1){bt=false;break;}if(vt=-1,bt){for(Qt=z(l.d7,H,6,32,15,1),Ht=0,i=0;i<t.f;i++)if((t.u[i]&3)!=0&&(t.u[i]&3)!=3&&(t.u[i]&U1)>>18==1){Lt=(t.u[i]&U1)>>18!=1&&(t.u[i]&U1)>>18!=2?-1:(t.u[i]&k0)>>20,++Qt[Lt],Ht<Qt[Lt]&&(Ht=Qt[Lt],vt=Lt);break;}}for(this.a=A||new T.Pjb(),Ot=t.P!=null?t.P:"",T.Njb(this.a,Ot+Nt),T.Njb(this.a,"Actelion Java MolfileCreator 1.0"+Nt+Nt),n.Yr(this,t.q),n.Yr(this,t.r),T.Njb(this.a,"  0  0"),n.Yr(this,bt?0:1),T.Njb(this.a,"  0  0  0  0  0999 V2000"+Nt),ot=t.q==1,Q=1;Q<t.q;Q++)if(t.J[Q].a!=t.J[0].a||t.J[Q].b!=t.J[0].b||t.J[Q].c!=t.J[0].c){ot=true;break;}if(Dt=1,ot)if(e!=0)Dt=e;else if(v=n.Vl(t,t.q,t.r,(n.Qk(),n.Pk)),v!=0)(v<1||v>3)&&(Dt=1.5/v);else {for(xt=L0,i=1;i<t.q;i++)for(o=0;o<i;o++)N=t.J[o].a-t.J[i].a,I=t.J[o].b-t.J[i].b,ut=t.J[o].c-t.J[i].c,K=N*N+I*I+ut*ut,xt>K&&(xt=K);Dt=3/xt;}for(L=0;L<t.q;L++)ot?(n.Xr(this,Dt*t.J[L].a),n.Xr(this,Dt*-t.J[L].b),n.Xr(this,Dt*-t.J[L].c)):T.Njb(this.a,"    0.0000    0.0000    0.0000"),(t.v==null?null:t.v[L])!=null?T.Njb(this.a," L  "):kt(G(t.B[L],1),0)?T.Njb(this.a," A  "):t.C[L]>=129&&t.C[L]<=144||t.C[L]==154?T.Njb(this.a," R# "):(d=(n.Qk(),n.Kk)[t.C[L]],T.Njb(this.a," "+d),w.Xzb(d).length==1?T.Njb(this.a,"  "):w.Xzb(d).length==2&&T.Njb(this.a," ")),T.Njb(this.a," 0  0  0"),rt=G(Z0,t.B[L]),et(rt,0)==0?T.Njb(this.a,"  0"):et(rt,384)==0?T.Njb(this.a,"  3"):et(rt,128)==0?T.Njb(this.a,"  2"):et(rt,1792)==0?T.Njb(this.a,"  1"):et(rt,1664)==0&&T.Njb(this.a,"  2"),T.Njb(this.a,kt(G(t.B[L],Q0),0)?"  1":"  0"),f1=((t.u[L]&be)>>>27)-1,f1==-1?T.Njb(this.a,"  0"):f1==0?T.Njb(this.a," 15"):n.Yr(this,f1),T.Njb(this.a,"  0  0  0"),n.Yr(this,j.Math.abs(t.w[L])),T.Njb(this.a,"  0  0"+Nt);for(M=0;M<t.r;M++){switch(t.H[M]){case 1:mt=1,Zt=0;break;case 2:mt=2,Zt=0;break;case 4:mt=3,Zt=0;break;case 129:mt=1,Zt=6;break;case 257:mt=1,Zt=1;break;case 386:mt=2,Zt=3;break;case 8:mt=4,Zt=0;break;case 16:mt=8,Zt=0;break;default:mt=1,Zt=0;}bt&&(Zt==1||Zt==6)&&(i=t.D[0][M],(t.u[i]&U1)>>18==2?Zt=0:(t.u[i]&U1)>>18==1&&((t.u[i]&U1)>>18!=1&&(t.u[i]&U1)>>18!=2?-1:(t.u[i]&k0)>>20)!=vt&&(Zt=4)),S=t.G[M]&31,S!=0&&(S==8?mt=4:S==3?mt=5:S==9?mt=6:S==10?mt=7:mt=8),u1=t.G[M]&384,Et=u1==0?0:u1==256?1:2,n.Yr(this,1+t.D[0][M]),n.Yr(this,1+t.D[1][M]),n.Yr(this,mt),n.Yr(this,Zt),T.Njb(this.a,"  0"),n.Yr(this,Et),T.Njb(this.a,"  0"+Nt);}for(pt=0,s=0;s<t.q;s++)t.s[s]!=0&&++pt;if(pt!=0)for(O=0,i=0;i<t.q;i++)t.s[i]!=0&&(O==0&&(T.Njb(this.a,"M  CHG"),n.Yr(this,j.Math.min(8,pt))),T.Njb(this.a," "),n.Yr(this,i+1),V=t.s[i],V<0?(T.Njb(this.a,"  -"),V=-V):T.Njb(this.a,"   "),T.Jjb(this.a,48+V&r1),--pt,(++O==8||pt==0)&&(O=0,T.Njb(this.a,Nt)));for(pt=0,h=0;h<t.q;h++)t.A[h]==0||++pt;if(pt!=0)for(O=0,i=0;i<t.q;i++)t.A[i]!=0&&(O==0&&(T.Njb(this.a,"M  ISO"),n.Yr(this,j.Math.min(8,pt))),T.Njb(this.a," "),n.Yr(this,i+1),T.Njb(this.a," "),n.Yr(this,t.A[i]),--pt,(++O==8||pt==0)&&(O=0,T.Njb(this.a,Nt)));for(pt=0,b=0;b<t.q;b++)(t.u[b]&48)!=0&&++pt;if(pt!=0){for(O=0,i=0;i<t.q;i++)if((t.u[i]&48)!=0){switch(O==0&&(T.Njb(this.a,"M  RAD"),n.Yr(this,j.Math.min(8,pt))),T.Njb(this.a," "),n.Yr(this,i+1),t.u[i]&48){case 16:T.Njb(this.a,"   1");break;case 32:T.Njb(this.a,"   2");break;case 48:T.Njb(this.a,"   3");}--pt,(++O==8||pt==0)&&(O=0,T.Njb(this.a,Nt));}}for(pt=0,c=0;c<t.q;c++)(t.C[c]>=129&&t.C[c]<=144||t.C[c]==154)&&++pt;if(pt!=0)for(O=0,i=0;i<t.q;i++)C=t.C[i],(C>=129&&C<=144||C==154)&&(O==0&&(T.Njb(this.a,"M  RGP"),n.Yr(this,j.Math.min(8,pt))),T.Njb(this.a," "),n.Yr(this,i+1),T.Njb(this.a," "),n.Yr(this,C==154?0:C>=142?C-141:C-125),--pt,(++O==8||pt==0)&&(O=0,T.Njb(this.a,Nt)));if(t.K){for(pt=0,i=0;i<t.q;i++)kt(G(t.B[i],120),0)&&++pt;if(pt!=0)for(O=0,o=0;o<t.q;o++)y1=G(t.B[o],120),et(y1,0)!=0&&(O==0&&(T.Njb(this.a,"M  RBC"),n.Yr(this,j.Math.min(8,pt))),T.Njb(this.a," "),n.Yr(this,o+1),et(y1,112)==0?T.Njb(this.a,"  -1"):et(y1,8)==0?T.Njb(this.a,"   1"):et(y1,104)==0?T.Njb(this.a,"   2"):et(y1,88)==0?T.Njb(this.a,"   3"):et(y1,56)==0&&T.Njb(this.a,"   4"),--pt,(++O==8||pt==0)&&(O=0,T.Njb(this.a,Nt)));for(g=0;g<t.q;g++)if(m=t.v==null?null:t.v[g],m!=null){for(T.Njb(this.a,"M  ALS "),n.Yr(this,g+1),n.Yr(this,m.length),T.Njb(this.a,kt(G(t.B[g],1),0)?" T ":" F "),yt=0;yt<m.length;yt++)switch(it=(n.Qk(),n.Kk)[m[yt]],w.Xzb(it).length){case 1:T.Njb(this.a,it+"   ");break;case 2:T.Njb(this.a,it+"  ");break;case 3:T.Njb(this.a,it+" ");break;default:T.Njb(this.a,"   ?");}T.Njb(this.a,Nt);}for(pt=0,D=0;D<t.q;D++)kt(G(t.B[D],6144),0)&&++pt;if(pt!=0)for(O=0,o=0;o<t.q;o++)m1=G(t.B[o],6144),et(m1,0)!=0&&(O==0&&(T.Njb(this.a,"M  SUB"),n.Yr(this,j.Math.min(8,pt))),T.Njb(this.a," "),n.Yr(this,o+1),kt(G(m1,Y1),0)?T.Njb(this.a,"   "+(t.e[o]+1)):T.Njb(this.a,"  -2"),--pt,(++O==8||pt==0)&&(O=0,T.Njb(this.a,Nt)));}for(pt=0,u=0;u<t.q;u++)(t.t==null||t.t[u]==null?null:T.cjb(($t=t.t[u],V1.Hmb(),$t)))!=null&&++pt;if(pt!=0)for(Mt=0,P=0;P<t.q;P++)it=t.t==null||t.t[P]==null?null:T.cjb((ct=t.t[P],V1.Hmb(),ct)),it!=null&&(++Mt,T.Njb(this.a,"M  STY  1 "),n.Yr(this,Mt),T.Njb(this.a," DAT"),T.Njb(this.a,Nt),T.Njb(this.a,"M  SLB  1 "),n.Yr(this,Mt),T.Njb(this.a," "),n.Yr(this,Mt),T.Njb(this.a,Nt),T.Njb(this.a,"M  SAL "),n.Yr(this,Mt),T.Njb(this.a,"  1 "),n.Yr(this,P+1),T.Njb(this.a,Nt),T.Njb(this.a,"M  SDT "),n.Yr(this,Mt),T.Njb(this.a," "),T.Njb(this.a,iP),T.Njb(this.a,Nt),T.Njb(this.a,"M  SDD "),n.Yr(this,Mt),T.Njb(this.a," "),n.Xr(this,t.J[P].a),n.Xr(this,t.J[P].b),T.Njb(this.a,"    DA    ALL  1       5"),T.Njb(this.a,Nt),T.Njb(this.a,"M  SED "),n.Yr(this,Mt),T.Njb(this.a," "),T.Njb(this.a,it),T.Njb(this.a,Nt));T.Njb(this.a,me+Nt);},n.bs=function(t,e){n.as.call(this,t,0,e);},x(112,1,{},n.$r,n.as),l.M7=X(112),n.cs=function(t,e,A,P,r,i){var D,u,o,Q;switch(Q=1,u=false,r){case 1:Q=257;break;case 3:Q=386;break;case 4:Q=257,u=true;break;case 6:Q=129;break;default:switch(P){case 1:Q=1;break;case 2:Q=2;break;case 3:Q=4;break;case 4:Q=8;break;case 9:Q=16;}}if(D=n.Vk(t.j,e,A,Q),o=0,u&&n.nn(t.j,e,1,-1),P>4)switch(P){case 5:o|=3;break;case 6:o|=9;break;case 7:o|=10;break;case 8:Q!=16&&(o|=127);}return i==1&&(o|=256),i==2&&(o|=128),o!=0&&n.Kn(t.j,D,o,true),D;},n.ds=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(D=z(l.d7,H,6,t.j.q,15,1),A=0;A<t.j.q;A++)D[A]=-n.Al(t.j,A);for(n.Fe(new n.Ue(t.j),null,true),P=0;P<t.j.q;P++)D[P]+=n.Al(t.j,P);for(e=0;e<t.j.q;e++)if(D[e]!=0)for(u=-D[e],r=0;r<t.j.r;r++)for(Q=0;Q<2;Q++)u>0&&n.hm(t.j,r)==16&&n.Yl(t.j,1-Q,r)==e&&(s=n.Yl(t.j,Q,r),n.Km(t.j,s)&&(L=n.fs(t,s),i=n.Al(t.j,s),i<L&&(o=j.Math.min(u,L-i),n.hn(t.j,s,i+o),u-=o)));},n.es=function(t,e){return t.j=null,n.zs(t,new Rt.igb(new Rt.tgb(e)))?t.j:null;},n.fs=function(t,e){var A,P;return A=n.Sl(t.j,e),P=A<(n.Qk(),n.Mk).length?n.Mk[A]:null,P==null?0:P[P.length-1];},n.gs=function(t,e){var A;return A=t.b?U.jpb(t.b,T.uib(e)):null,A?A.a:e-1;},n.hs=function(t,e){var A;return A=t.c?U.jpb(t.c,T.uib(e)):null,A?A.a:e-1;},n.is=function(t,e){var A,P;if(e!=null)for(n.aw(t.j,1),A=0;A<t.j.f;A++)e[A]!=0&&(P=n.lm(t.j,A,n.up(t.j,A)),e[A]==15?P>=0&&n.fn(t.j,A,0):e[A]!=n.nm(t.j,A)&&n.fn(t.j,A,e[A]-P));},n.js=function(t,e){var A,P;for(A=e;A<w.Xzb(t).length;A++){if(P=(w.Szb(A,w.Xzb(t).length),w.Xzb(t).charCodeAt(A)),P==32||P==9)return  -1;if(P==61)return A;}return  -1;},n.ks=function(t,e){var A;if(e==-1)return  -1;for(A=e+1;A<w.Xzb(t).length;A++)if(w.Szb(A,w.Xzb(t).length),w.Xzb(t).charCodeAt(A)!=32&&(w.Szb(A,w.Xzb(t).length),w.Xzb(t).charCodeAt(A)!=9))return A;return  -1;},n.ls=function(t,e){var A,P,r,i;for(A=false,r=false,P=e;P<w.Xzb(t).length;P++)if(i=(w.Szb(P,w.Xzb(t).length),w.Xzb(t).charCodeAt(P)),i==40?A=true:i==41?A=false:i==34&&(r=!r),!A&&!r&&(i==32||i==9))return P;return  -1;},n.ms=function(t){return w.Xzb(t).indexOf("ATOMS=(")!=-1?qn:w.Xzb(t).indexOf("BONDS=(")!=-1?"BONDS":null;},n.ns=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut;for(g=0,d=(u=n.ls(e,g),u==-1?w.Xzb(e).length:u),P=T.Qhb((w.Rzb(g,d,w.Xzb(e).length),w.Xzb(e).substr(g,d-g)),P1,St),g=n.ks(e,d),d=(o=n.ls(e,g),o==-1?w.Xzb(e).length:o),C=(w.Rzb(g,d,w.Xzb(e).length),w.Xzb(e).substr(g,d-g)),O=null,r=false,m=n.vs(e),m!=0&&(O=n.os(e),m<0&&(r=true),d=j.Math.abs(m)),g=n.ks(e,d),d=(Q=n.ls(e,g),Q==-1?w.Xzb(e).length:Q),N=I0((w.Rzb(g,d,w.Xzb(e).length),w.Xzb(e).substr(g,d-g))),g=n.ks(e,d),d=(L=n.ls(e,g),L==-1?w.Xzb(e).length:L),I=I0((w.Rzb(g,d,w.Xzb(e).length),w.Xzb(e).substr(g,d-g))),g=n.ks(e,d),d=(s=n.ls(e,g),s==-1?w.Xzb(e).length:s),ut=I0((w.Rzb(g,d,w.Xzb(e).length),w.Xzb(e).substr(g,d-g))),g=n.ks(e,d),d=(h=n.ls(e,g),h==-1?w.Xzb(e).length:h),M=T.Qhb((w.Rzb(g,d,w.Xzb(e).length),w.Xzb(e).substr(g,d-g)),P1,St),A=n.Sk(t.j,N,-I,-ut),A+1!=P&&(!t.b&&(t.b=new U.awb()),U.Uvb(t.b,T.uib(P),T.uib(A))),O!=null&&(n.Bn(t.j,A,6),n.pn(t.j,A,O,r)),M!=0&&n.qn(t.j,A,M,false),T.ejb(C,"A")||T.ejb(C,"*")?n.un(t.j,A,1,true):T.ejb(C,"Q")?(v=z(l.d7,H,6,1,15,1),v[0]=6,n.pn(t.j,A,v,true)):n.Bn(t.j,A,n.oo(C,67));(g=n.ks(e,d))!=-1;)if(d=(D=n.ls(e,g),D==-1?w.Xzb(e).length:D),S=(w.Rzb(g,d,w.Xzb(e).length),w.Xzb(e).substr(g,d-g)),c=T.ljb(S,Z1(61)),b=(w.Rzb(0,c,w.Xzb(S).length),w.Xzb(S).substr(0,c)),K=T.Qhb((w.Szb(c+1,w.Xzb(S).length+1),w.Xzb(S).substr(c+1)),P1,St),T.ejb(b,"CHG"))n.hn(t.j,A,K);else if(T.ejb(b,"RAD"))switch(K){case 1:n.vn(t.j,A,16);break;case 2:n.vn(t.j,A,32);break;case 3:n.vn(t.j,A,48);}else if(!T.ejb(b,"CFG")){if(T.ejb(b,"MASS"))n.sn(t.j,A,K);else if(T.ejb(b,"VAL"))n.fn(t.j,A,K==-1?0:K==0?-1:K);else if(T.ejb(b,"HCOUNT"))switch(K){case 0:break;case  -1:n.un(t.j,A,1792,true);break;case 1:n.un(t.j,A,128,true);break;case 2:n.un(t.j,A,384,true);break;default:n.un(t.j,A,896,true);}else if(T.ejb(b,"SUBST")){if(K==-1)n.un(t.j,A,z1,true);else if(K>0){for(V=0,i=0;i<t.j.r;i++)(n.Yl(t.j,0,i)==A||n.Yl(t.j,1,i)==A)&&++V;K>V&&n.un(t.j,A,Y1,true);}}else if(T.ejb(b,"RBCNT"))switch(K){case 3:case  -1:n.un(t.j,A,112,true);break;case 1:n.un(t.j,A,8,true);break;case 2:n.un(t.j,A,104,true);break;case 4:n.un(t.j,A,56,true);}}},n.os=function(t){var e,A,P,r,i,D,u,o;if(u=null,A=w.Xzb(t).indexOf("["),P=w.Xzb(t).indexOf("]",A),A>=0&&P>0){for(e=z(l.d7,H,6,16,15,1),o=(w.Rzb(A+1,P,w.Xzb(t).length),w.Xzb(t).substr(A+1,P-(A+1))),r=0,D=true;D&&r<16;)A=w.Xzb(o).indexOf(","),i=null,A==-1?(i=o,D=false):(i=(w.Rzb(0,A,w.Xzb(o).length),w.Xzb(o).substr(0,A)),o=(w.Szb(A+1,w.Xzb(o).length+1),w.Xzb(o).substr(A+1))),e[r++]=n.oo(i,1);u=z(l.d7,H,6,r,15,1),T.Vjb(e,0,u,0,r);}return u;},n.ps=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v;for(c=0,g=(o=n.ls(e,c),o==-1?w.Xzb(e).length:o),i=T.Qhb((w.Rzb(c,g,w.Xzb(e).length),w.Xzb(e).substr(c,g-c)),P1,St),c=n.ks(e,g),g=(Q=n.ls(e,c),Q==-1?w.Xzb(e).length:Q),D=T.Qhb((w.Rzb(c,g,w.Xzb(e).length),w.Xzb(e).substr(c,g-c)),P1,St),c=n.ks(e,g),g=(L=n.ls(e,c),L==-1?w.Xzb(e).length:L),A=n.gs(t,T.Qhb((w.Rzb(c,g,w.Xzb(e).length),w.Xzb(e).substr(c,g-c)),P1,St)),c=n.ks(e,g),g=(s=n.ls(e,c),s==-1?w.Xzb(e).length:s),P=n.gs(t,T.Qhb((w.Rzb(c,g,w.Xzb(e).length),w.Xzb(e).substr(c,g-c)),P1,St)),m=0,C=0;(c=n.ks(e,g))!=-1;)if(g=(u=n.ls(e,c),u==-1?w.Xzb(e).length:u),d=(w.Rzb(c,g,w.Xzb(e).length),w.Xzb(e).substr(c,g-c)),b=T.ljb(d,Z1(61)),h=(w.Rzb(0,b,w.Xzb(d).length),w.Xzb(d).substr(0,b)),v=T.Qhb((w.Szb(b+1,w.Xzb(d).length+1),w.Xzb(d).substr(b+1)),P1,St),T.ejb(h,"CFG"))switch(v){case 1:m=1;break;case 2:m=D==2?3:4;break;case 3:m=6;}else T.ejb(h,"TOPO")&&(C=v);r=n.cs(t,A,P,D,m,C),r+1!=i&&(!t.c&&(t.c=new U.awb()),U.Uvb(t.c,T.uib(i),T.uib(r)));},n.qs=function(t,e){var A,P,r,i,D,u;if(u=n.ms(e),u!=null){if(D=n.ss(e,u),T.ejb(w.Xzb(e).substr(0,13),"MDLV30/STEABS")){if(T.ejb(u,qn))for(i=0;i<D.length;i++)n.nn(t.j,n.gs(t,D[i]),0,-1);else for(r=0;r<D.length;r++)n.Fn(t.j,n.hs(t,D[r]),0,-1);}else if(T.ejb(w.Xzb(e).substr(0,13),"MDLV30/STERAC")){if(P=T.Qhb(T.vjb(e,13,n.ls(e,13)),P1,St),T.ejb(u,qn))for(i=0;i<D.length;i++)n.nn(t.j,n.gs(t,D[i]),1,P-1);else for(r=0;r<D.length;r++)n.Fn(t.j,n.hs(t,D[r]),1,P-1);}else if(T.ejb(w.Xzb(e).substr(0,13),"MDLV30/STEREL")){if(P=T.Qhb(T.vjb(e,13,n.ls(e,13)),P1,St),T.ejb(u,qn))for(i=0;i<D.length;i++)n.nn(t.j,n.gs(t,D[i]),2,P-1);else for(r=0;r<D.length;r++)n.Fn(t.j,n.hs(t,D[r]),2,P-1);}else if(T.ejb(w.Xzb(e).substr(0,13),"MDLV30/HILITE"))if(T.ejb(u,qn))for(r=0;r<D.length;r++)n.jn(t.j,n.gs(t,D[r]),448);else for(r=0;r<D.length;r++)A=n.hs(t,D[r]),n.jn(t.j,n.Yl(t.j,0,A),448),n.jn(t.j,n.Yl(t.j,1,A),448);}},n.rs=function(t,e){var A,P,r;t.j||T.ejb(w.Xzb(e).substr(0,6),"COUNTS")&&(A=n.ks(e,n.ls(e,7)),P=T.Qhb(T.vjb(e,7,n.ls(e,7)),P1,St),r=T.Qhb(T.vjb(e,A,n.ls(e,A)),P1,St),t.j=new n.tw(P,r));},n.ss=function(t,e){var A,P,r,i,D,u;for(i=w.Xzb(t).indexOf(e+"=(")+w.Xzb(e).length+2,D=T.mjb(t,Z1(41),i),r=n.ls(t,i),A=T.Qhb((w.Rzb(i,r,w.Xzb(t).length),w.Xzb(t).substr(i,r-i)),P1,St),u=z(l.d7,H,6,A,15,1),P=0;P<A;P++)i=n.ks(t,r),r=n.ls(t,i),(r==-1||r>D)&&(r=D),u[P]=T.Qhb((w.Rzb(i,r,w.Xzb(t).length),w.Xzb(t).substr(i,r-i)),P1,St);return u;},n.ts=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v;if(h=0,b=(i=n.ls(e,h),i==-1?w.Xzb(e).length:i),v=(w.Rzb(h,b,w.Xzb(e).length),w.Xzb(e).substr(h,b-h)),T.ejb("DEFAULT",v)&&(h=n.ks(e,b),b=(D=n.ls(e,h),D==-1?w.Xzb(e).length:D),v=(w.Rzb(h,b,w.Xzb(e).length),w.Xzb(e).substr(h,b-h)),T.ejb(w.Xzb(v).substr(0,5),"CLASS")&&(h=n.ks(e,b),b=(u=n.ls(e,h),u==-1?w.Xzb(e).length:u),v=(w.Rzb(h,b,w.Xzb(e).length),w.Xzb(e).substr(h,b-h)),T.ejb("-",v)&&(h=n.ks(e,b),b=(o=n.ls(e,h),o==-1?w.Xzb(e).length:o),v=(w.Rzb(h,b,w.Xzb(e).length),w.Xzb(e).substr(h,b-h))))),T.Qhb(v,P1,St),h=n.ks(e,b),b=(Q=n.ls(e,h),Q==-1?w.Xzb(e).length:Q),C=(w.Rzb(h,b,w.Xzb(e).length),w.Xzb(e).substr(h,b-h)),!!T.ejb("DAT",C)){for(h=n.ks(e,b),b=(r=n.ls(e,h),r==-1?w.Xzb(e).length:r),T.Qhb((w.Rzb(h,b,w.Xzb(e).length),w.Xzb(e).substr(h,b-h)),P1,St),m=new U.awb(),h=n.ks(e,b);h!=-1;)h=n.ys(m,e,h);P=(g=qn,U.anb(U.Pvb(m,g))),s=(d="FIELDNAME",U.anb(U.Pvb(m,d))),L=(c="FIELDDATA",U.anb(U.Pvb(m,c))),!(!T.ejb(iP,s)||P==null||L==null)&&(A=T.sjb(P," "),A.length==2&&n.ln(t.j,T.Qhb(A[1],P1,St)-1,L));}},n.us=function(t){var e,A,P,r,i,D,u,o,Q;for(o=z(l.d7,H,6,t.j.q,15,1),P=0;P<t.j.r;P++)if(n.hm(t.j,P)==8)for(D=0;D<2;D++)o[n.Yl(t.j,D,P)]=1;for(r=0;r<t.j.r;r++)for(Q=n.em(t.j,r),i=0;i<2;i++)o[n.Yl(t.j,i,r)]+=Q;for(A=0;A<t.j.r;A++)if(n.em(t.j,A)==1){for(i=0;i<2;i++)if(u=n.Yl(t.j,1-i,A),n.Km(t.j,u)&&(e=n.Yl(t.j,i,A),n.Gm(t.j,e)&&o[e]>n.nm(t.j,e))){n.Ln(t.j,A,16);continue;}}},n.vs=function(t){var e,A;if(w.Xzb(t).indexOf("[")>=0){if(e=w.Xzb(t).indexOf(" NOT["),A=w.Xzb(t).indexOf("]",e),e>=0&&A>0)return -(A+1);if(e=w.Xzb(t).indexOf(" ["),A=w.Xzb(t).indexOf("]",e),e>=0&&A>0)return A+1;if(e=w.Xzb(t).indexOf(" 'NOT["),A=w.Xzb(t).indexOf("]'",e),e>=0&&A>0)return -(A+2);if(e=w.Xzb(t).indexOf(" '["),A=w.Xzb(t).indexOf("]'",e),e>=0&&A>0)return A+2;T.Ujb();}return 0;},n.ws=function(t,e,A){return t.j=e,n.zs(t,A);},n.xs=function(t){return w.Xzb(t).length==0?0:T.Qhb(t,P1,St);},n.ys=function(t,e,A){var P,r,i,D,u,o,Q;return i=n.js(e,A),i==-1?-1:(r=(P=n.ls(e,i+1),P==-1?w.Xzb(e).length:P),D=(w.Rzb(A,i,w.Xzb(e).length),w.Xzb(e).substr(A,i-A)),Q=(w.Rzb(i+1,r,w.Xzb(e).length),w.Xzb(e).substr(i+1,r-(i+1))),(T.ejb(w.Xzb(Q).substr(0,1),"(")&&(o=w.Xzb(")").length,T.ejb(w.Xzb(Q).substr(w.Xzb(Q).length-o,o),")"))||T.ejb(w.Xzb(Q).substr(0,1),'"')&&(u=w.Xzb('"').length,T.ejb(w.Xzb(Q).substr(w.Xzb(Q).length-u,u),'"')))&&(Q=T.vjb(Q,1,w.Xzb(Q).length-1)),U.Uvb(t,D,Q),n.ks(e,r));},n.zs=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt,mt,y1,u1,Mt,Zt,m1,Et;u1=null,c=null;try{if(t.f=null,t.j&&(n.$v(t.j),n.Nn(t.j,!1)),Lt=N=Rt.hgb(e),Lt==null||(N=Rt.hgb(e))==null||(N=Rt.hgb(e))==null||(t.k=w.Xzb(N).indexOf("From CSD data. Using bond type 'Any'")!=-1,t.e=w.Xzb(N).indexOf("From CSD data.")!=-1,(N=Rt.hgb(e))==null))return !1;t.g=!1,t.d=t.a;try{ot=T.Qhb(T.wjb((w.Rzb(0,3,w.Xzb(N).length),w.Xzb(N).substr(0,3))),P1,St),rt=T.Qhb(T.wjb((w.Rzb(3,6,w.Xzb(N).length),w.Xzb(N).substr(3,3))),P1,St),yt=n.xs(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3)))),t.d=t.d|n.xs(T.wjb((w.Rzb(12,15,w.Xzb(N).length),w.Xzb(N).substr(12,3))))==1,t.g=w.Xzb(N).length>=39&&T.ejb(w.Xzb(N).substr(34,w.Xzb("V3000").length),"V3000");}catch(f1){if(f1=H1(f1),Jt(f1,19))return !1;throw Ut(f1);}if(t.g)return wt=n.As(t,e),n.Tn(t.j,Lt),wt;if(!t.j&&(t.j=new n.tw(ot,rt)),n.Tn(t.j,Lt),t.d||(t.j.L=!0),ot==0){for(;N!=null&&!(T.ejb(N,me)||T.ejb(N,QA)||T.ejb((w.Szb(1,w.Xzb(N).length+1),w.Xzb(N).substr(1)),"$"));)N=Rt.hgb(e);return !0;}for(v=0;v<ot;v++){if((N=Rt.hgb(e))==null)return !1;switch(Zt=I0(T.wjb((w.Rzb(0,10,w.Xzb(N).length),w.Xzb(N).substr(0,10)))),m1=I0(T.wjb((w.Rzb(10,20,w.Xzb(N).length),w.Xzb(N).substr(10,10)))),Et=I0(T.wjb((w.Rzb(20,30,w.Xzb(N).length),w.Xzb(N).substr(20,10)))),P=n.Sk(t.j,Zt,-m1,-Et),K=T.wjb((w.Rzb(31,34,w.Xzb(N).length),w.Xzb(N).substr(31,3))),T.ejb(K,"A")||T.ejb(K,"*")?n.un(t.j,P,1,!0):T.ejb(K,"Q")?(I=z(l.d7,H,6,1,15,1),I[0]=6,n.pn(t.j,P,I,!0)):(D=n.oo(K,67),n.Bn(t.j,P,D)),Dt=n.xs(T.wjb((w.Rzb(34,36,w.Xzb(N).length),w.Xzb(N).substr(34,2)))),Dt!=0&&n.sn(t.j,P,(n.Qk(),n.Ok)[n.Sl(t.j,P)]+Dt),s=n.xs(T.wjb((w.Rzb(36,39,w.Xzb(N).length),w.Xzb(N).substr(36,3)))),s!=0&&(s==4?n.vn(t.j,P,32):n.hn(t.j,P,4-s)),ut=w.Xzb(N).length<63?0:n.xs(T.wjb((w.Rzb(60,63,w.Xzb(N).length),w.Xzb(N).substr(60,3)))),n.qn(t.j,P,ut,!1),m=w.Xzb(N).length<45?0:n.xs(T.wjb((w.Rzb(42,45,w.Xzb(N).length),w.Xzb(N).substr(42,3)))),m){case 0:break;case 1:n.un(t.j,P,768,!0);break;case 2:n.un(t.j,P,128,!0);break;case 3:n.un(t.j,P,384,!0);break;default:n.un(t.j,P,896,!0);}w.Xzb(N).length>=48&&(w.Szb(47,w.Xzb(N).length),w.Xzb(N).charCodeAt(47)==49)&&n.un(t.j,P,Q0,!0),y1=w.Xzb(N).length<51?0:n.xs(T.wjb((w.Rzb(48,51,w.Xzb(N).length),w.Xzb(N).substr(48,3)))),y1!=0&&(u1==null&&(u1=z(l.d7,H,6,ot,15,1)),u1[P]=y1);}for(M=0;M<rt;M++){if((N=Rt.hgb(e))==null)return !1;r=T.Qhb(T.wjb((w.Rzb(0,3,w.Xzb(N).length),w.Xzb(N).substr(0,3))),P1,St)-1,i=T.Qhb(T.wjb((w.Rzb(3,6,w.Xzb(N).length),w.Xzb(N).substr(3,3))),P1,St)-1,Q=T.Qhb(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3))),P1,St),xt=w.Xzb(N).length<12?0:n.xs(T.wjb((w.Rzb(9,12,w.Xzb(N).length),w.Xzb(N).substr(9,3)))),mt=w.Xzb(N).length<18?0:n.xs(T.wjb((w.Rzb(15,18,w.Xzb(N).length),w.Xzb(N).substr(15,3)))),Q==8&&(t.k||n.Km(t.j,r)||n.Km(t.j,i))&&(Q=9),n.cs(t,r,i,Q,xt,mt);}for(C=0;C<yt;C++)if((N=Rt.hgb(e))==null)return !1;if((N=Rt.hgb(e))==null)return (t.i&1)!=0&&(t.f=n.kp(t.j)),n.is(t,u1),t.d||n.aw(t.j,15),!0;for(;N!=null&&!(T.ejb(N,me)||T.ejb(N,QA));){if(T.ejb(w.Xzb(N).substr(0,6),"M  CHG")&&(S=T.Qhb(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3))),P1,St),S>0))for(A=10,Mt=14,V=1;V<=S;++V,A+=8,Mt+=8)P=T.Qhb(T.wjb((w.Rzb(A,A+3,w.Xzb(N).length),w.Xzb(N).substr(A,A+3-A))),P1,St)-1,L=T.Qhb(T.wjb((w.Rzb(Mt,Mt+3,w.Xzb(N).length),w.Xzb(N).substr(Mt,Mt+3-Mt))),P1,St),n.hn(t.j,P,L);if(T.ejb(w.Xzb(N).substr(0,6),"M  ISO")&&(S=T.Qhb(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3))),P1,St),S>0))for(A=10,Mt=14,V=1;V<=S;++V,A+=8,Mt+=8)P=T.Qhb(T.wjb((w.Rzb(A,A+3,w.Xzb(N).length),w.Xzb(N).substr(A,A+3-A))),P1,St)-1,Qt=T.Qhb(T.wjb((w.Rzb(Mt,Mt+3,w.Xzb(N).length),w.Xzb(N).substr(Mt,Mt+3-Mt))),P1,St),n.sn(t.j,P,Qt);if(T.ejb(w.Xzb(N).substr(0,6),"M  RAD")&&(S=T.Qhb(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3))),P1,St),S>0))for(A=10,Mt=14,V=1;V<=S;++V,A+=8,Mt+=8)switch(P=T.Qhb(T.wjb((w.Rzb(A,A+3,w.Xzb(N).length),w.Xzb(N).substr(A,A+3-A))),P1,St)-1,ct=T.Qhb(T.wjb((w.Rzb(Mt,Mt+3,w.Xzb(N).length),w.Xzb(N).substr(Mt,Mt+3-Mt))),P1,St),ct){case 1:n.vn(t.j,P,16);break;case 2:n.vn(t.j,P,32);break;case 3:n.vn(t.j,P,48);}if((T.ejb(w.Xzb(N).substr(0,6),"M  RBC")||T.ejb(w.Xzb(N).substr(0,6),"M  RBD"))&&(S=T.Qhb(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3))),P1,St),S>0))for(A=10,Mt=14,V=1;V<=S;++V,A+=8,Mt+=8)switch(P=T.Qhb(T.wjb((w.Rzb(A,A+3,w.Xzb(N).length),w.Xzb(N).substr(A,A+3-A))),P1,St)-1,$t=T.Qhb(T.wjb((w.Rzb(Mt,Mt+3,w.Xzb(N).length),w.Xzb(N).substr(Mt,Mt+3-Mt))),P1,St),$t){case 3:case -1:n.un(t.j,P,112,!0);break;case 1:n.un(t.j,P,8,!0);break;case 2:n.un(t.j,P,104,!0);break;case 4:n.un(t.j,P,56,!0);}if(T.ejb(w.Xzb(N).substr(0,6),"M  ALS")&&(P=T.Qhb(T.wjb((w.Rzb(7,10,w.Xzb(N).length),w.Xzb(N).substr(7,3))),P1,St)-1,P>=0)){for(bt=T.Qhb(T.wjb((w.Rzb(10,13,w.Xzb(N).length),w.Xzb(N).substr(10,3))),P1,St),u=(w.Szb(14,w.Xzb(N).length),w.Xzb(N).charCodeAt(14)==84),y1=z(l.d7,H,6,bt,15,1),A=16,V=0;V<bt;++V,A+=4)pt=T.wjb((w.Rzb(A,A+4,w.Xzb(N).length),w.Xzb(N).substr(A,A+4-A))),y1[V]=n.oo(pt,1);n.Bn(t.j,P,6),n.pn(t.j,P,y1,u);}if(T.ejb(w.Xzb(N).substr(0,6),"M  SUB")&&(S=T.Qhb(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3))),P1,St),S>0)){for(A=10,Mt=14,V=1;V<=S;++V,A+=8,Mt+=8)if(P=T.Qhb(T.wjb((w.Rzb(A,A+3,w.Xzb(N).length),w.Xzb(N).substr(A,A+3-A))),P1,St)-1,Ot=T.Qhb(T.wjb((w.Rzb(Mt,Mt+3,w.Xzb(N).length),w.Xzb(N).substr(Mt,Mt+3-Mt))),P1,St),Ot==-2)n.un(t.j,P,z1,!0);else if(Ot>0){for(Nt=0,o=0;o<t.j.r;o++)(n.Yl(t.j,0,o)==P||n.Yl(t.j,1,o)==P)&&++Nt;Ot>Nt&&n.un(t.j,P,Y1,!0);}}if(T.ejb(w.Xzb(N).substr(0,6),"M  RGP")&&(S=T.Qhb(T.wjb((w.Rzb(6,9,w.Xzb(N).length),w.Xzb(N).substr(6,3))),P1,St),S>0))for(A=10,Mt=14,V=1;V<=S;++V,A+=8,Mt+=8)P=T.Qhb(T.wjb((w.Rzb(A,A+3,w.Xzb(N).length),w.Xzb(N).substr(A,A+3-A))),P1,St)-1,vt=T.Qhb(T.wjb((w.Rzb(Mt,Mt+3,w.Xzb(N).length),w.Xzb(N).substr(Mt,Mt+3-Mt))),P1,St),vt>=1&&vt<=20&&n.Bn(t.j,P,n.oo("R"+vt,2));(T.ejb(w.Xzb(N).substr(0,6),"M  SAL")&&w.Xzb(N).length>=17||T.ejb(w.Xzb(N).substr(0,6),"M  SDT")&&w.Xzb(N).length>=12||T.ejb(w.Xzb(N).substr(0,6),"M  SED")&&w.Xzb(N).length>=12)&&(!c&&(c=new U.awb()),Ht=(w.Rzb(7,10,w.Xzb(N).length),w.Xzb(N).substr(7,3)),h=(O=Ht,U.anb(U.Pvb(c,O))),h||(h=new n.Ds(),U.Uvb(c,Ht,h)),T.ejb(w.Xzb(N).substr(0,6),"M  SAL")&&T.ejb(w.Xzb(N).substr(10,w.Xzb("  1").length),"  1")?h.a=T.Qhb(T.wjb((w.Rzb(13,17,w.Xzb(N).length),w.Xzb(N).substr(13,4))),P1,St):T.ejb(w.Xzb(N).substr(0,6),"M  SDT")?h.b=T.tjb((w.Szb(11,w.Xzb(N).length+1),w.Xzb(N).substr(11)),iP):T.ejb(w.Xzb(N).substr(0,6),"M  SED")&&(h.c=T.wjb((w.Szb(11,w.Xzb(N).length+1),w.Xzb(N).substr(11))))),N=Rt.hgb(e);}}catch(f1){if(f1=H1(f1),Jt(f1,19))return g=f1,T.LB(g,(T.Ujb(),T.Tjb),""),false;throw Ut(f1);}if(c)for(b=(it=new U.Job(c).a.ad().Sb(),new U.Qob(it));b.a.cd();)h=(d=b.a.dd(),d.gd()),h.b&&h.a!=0&&h.c!=null&&n.ln(t.j,h.a-1,h.c);return t.e&&(n.us(t),n.ds(t)),(t.i&1)!=0&&(t.f=n.kp(t.j)),n.is(t,u1),n.aw(t.j,15),true;},n.As=function(t,e){var A,P,r,i,D;for(t.b&&U.Nvb(t.b),t.c&&U.Nvb(t.c),r=0,P=Rt.hgb(e);P!=null&&T.ejb(w.Xzb(P).substr(0,7),LA);){for(P=T.wjb((w.Szb(7,w.Xzb(P).length+1),w.Xzb(P).substr(7)));D=w.Xzb("-").length,T.ejb(w.Xzb(P).substr(w.Xzb(P).length-D,D),"-");){if(A=Rt.hgb(e),!T.ejb(w.Xzb(A).substr(0,7),LA))return  false;P=T.wjb(T.ajb(T.vjb(P,0,w.Xzb(P).length-1),(w.Szb(7,w.Xzb(A).length+1),w.Xzb(A).substr(7))));}if(T.ejb(w.Xzb(P).substr(0,5),"BEGIN")){if(i=T.wjb((w.Szb(6,w.Xzb(P).length+1),w.Xzb(P).substr(6))),T.ejb(w.Xzb(i).substr(0,4),"CTAB"))r=1;else if(T.ejb(w.Xzb(i).substr(0,4),"ATOM"))r=2;else if(T.ejb(w.Xzb(i).substr(0,4),"BOND"))r=3;else if(T.ejb(w.Xzb(i).substr(0,6),"SGROUP"))r=4;else if(T.ejb(w.Xzb(i).substr(0,10),"COLLECTION"))r=5;else return  false;}else if(T.ejb(w.Xzb(P).substr(0,3),"END"))r=0;else if(r==1)n.rs(t,P);else if(r==2)n.ns(t,P);else if(r==3)n.ps(t,P);else if(r==4)n.ts(t,P);else if(r==5)n.qs(t,P);else return  false;P=Rt.hgb(e);}for(;P!=null&&!(T.ejb(w.Xzb(P).substr(0,6),me)||T.ejb(P,QA));)P=Rt.hgb(e);return  true;},n.Bs=function(){this.i=0;},n.Cs=function(){this.i=1;},x(128,1,{},n.Bs,n.Cs),y.a=false,y.d=false,y.e=false,y.g=false,y.i=0,y.k=false,l.O7=X(128),n.Ds=function(){},x(224,1,{224:1},n.Ds),y.a=0,y.b=false,l.N7=X(224),n.Es=function(t){},n.Fs=function(t,e){var A,P;for(P=o0.Imb(t.a,e),A=w.Xzb(P).length;A<10;A++)T.Jjb(t.b,32);T.Njb(t.b,P);},n.Gs=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t;for(I=(T.Ujb(),p1),T.Njb(t.b,"M  V30 BEGIN CTAB"+I),T.Njb(t.b,Y8+e.q+" "+e.r+" 0 0 0"+I),T.Njb(t.b,"M  V30 BEGIN ATOM"+I),o=0;o<e.q;o++){if(T.Njb(t.b,LA+(o+1)),(e.v==null?null:e.v[o])!=null){for(h=e.v==null?null:e.v[o],ut=kt(G(e.B[o],1),0),T.Njb(t.b,ut?" NOT[":" ["),V=0;V<h.length;V++)switch(V>0&&T.Njb(t.b,","),O=(n.Qk(),n.Kk)[h[V]],w.Xzb(O).length){case 2:case 3:case 1:T.Njb(t.b,O);break;default:T.Njb(t.b,"?");}T.Njb(t.b,"]");}else kt(G(e.B[o],1),0)?T.Njb(t.b," A"):e.C[o]>=129&&e.C[o]<=144||e.C[o]==154?T.Njb(t.b," R#"):T.Njb(t.b," "+(n.Qk(),n.Kk)[e.C[o]]);if(A?(T.Njb(t.b," "+Ft(Y0*t.c*e.J[o].a)/Y0),T.Njb(t.b," "+Ft(Y0*t.c*-e.J[o].b)/Y0),T.Njb(t.b," "+Ft(Y0*t.c*-e.J[o].c)/Y0)):T.Njb(t.b," 0 0 0"),T.Njb(t.b," "+j.Math.abs(e.w[o])),e.s[o]!=0&&T.Njb(t.b," CHG="+e.s[o]),(e.u[o]&48)!=0)switch(T.Njb(t.b," RAD="),e.u[o]&48){case 16:T.Njb(t.b,"1");break;case 32:T.Njb(t.b,"2");break;case 48:T.Njb(t.b,"3");}((e.u[o]&3)==1||(e.u[o]&3)==2)&&(T.Njb(t.b," CFG="),(e.u[o]&3)==1?T.Njb(t.b,"1"):T.Njb(t.b,"2")),e.A[o]!=0&&T.Njb(t.b," MASS="+e.A[o]),$t=((e.u[o]&be)>>>27)-1,$t!=-1&&T.Njb(t.b," VAL="+($t==0?"-1":T.uib($t))),b=e.C[o],(b>=129&&b<=144||b==154)&&T.Njb(t.b," RGROUPS=(1 "+(b==154?0:b>=142?b-141:b-125)+")"),S=G(Z0,e.B[o]),et(S,384)==0?T.Njb(t.b," HCOUNT=2"):et(S,128)==0?T.Njb(t.b," HCOUNT=1"):et(S,1792)==0?T.Njb(t.b," HCOUNT=-1"):et(S,1664)==0&&T.Njb(t.b," HCOUNT=1"),it=G(e.B[o],6144),et(it,0)!=0&&(kt(G(it,Y1),0)?T.Njb(t.b," SUBST="+(e.e[o]+1)):T.Njb(t.b," SUBST=-1")),rt=G(e.B[o],120),et(rt,0)!=0&&(et(rt,112)==0?T.Njb(t.b," RBCNT=-1"):et(rt,8)==0||et(rt,104)==0?T.Njb(t.b," RBCNT=2"):et(rt,88)==0?T.Njb(t.b," RBCNT=3"):et(rt,56)==0&&T.Njb(t.b," RBCNT=4")),T.Njb(t.b,I);}for(T.Njb(t.b,"M  V30 END ATOM"+I),T.Njb(t.b,"M  V30 BEGIN BOND"+I),g=0;g<e.r;g++){switch(T.Njb(t.b,LA+(g+1)),e.H[g]){case 1:Lt=1,bt=0;break;case 2:Lt=2,bt=0;break;case 4:Lt=3,bt=0;break;case 129:Lt=1,bt=3;break;case 257:Lt=1,bt=1;break;case 386:Lt=2,bt=2;break;case 8:Lt=4,bt=0;break;case 16:Lt=9,bt=0;break;default:Lt=1,bt=0;}m=e.G[g]&31,m!=0&&(m==8?Lt=4:m==3?Lt=5:m==9?Lt=6:m==10?Lt=7:Lt=8),T.Njb(t.b," "+Lt+" "+(e.D[0][g]+1)+" "+(e.D[1][g]+1)),bt!=0&&T.Njb(t.b," CFG="+bt),yt=e.G[g]&384,ct=yt==0?0:yt==256?1:2,ct!=0&&T.Njb(t.b," TOPO="+ct),T.Njb(t.b,I);}for(T.Njb(t.b,"M  V30 END BOND"+I),v=false,Q=0;Q<e.q;Q++)if((e.t==null||e.t[Q]==null?null:T.cjb((K=e.t[Q],V1.Hmb(),K)))!=null){v=true;break;}if(v){for(T.Njb(t.b,"M  V30 BEGIN SGROUP"+I),C=0,L=0;L<e.q;L++)(e.t==null||e.t[L]==null?null:T.cjb((N=e.t[L],V1.Hmb(),N)))!=null&&(++C,T.Njb(t.b,LA+C+" DAT "+C+" ATOMS=(1 "+(L+1)+') FIELDNAME="'+iP+'" -'+I),T.Njb(t.b,'M  V30 FIELDDISP="'),n.Fs(t,A?t.c*e.J[L].a:0),n.Fs(t,A?t.c*e.J[L].b:0),T.Njb(t.b,'    DA    ALL  1       5" FIELDDATA="'+(e.t==null||e.t[L]==null?null:T.cjb((K=e.t[L],V1.Hmb(),K)))+'"'+I));T.Njb(t.b,"M  V30 END SGROUP"+I);}for(ot=false,P=0,Qt=z(l.d7,H,6,32,15,1),i=z(l.d7,H,6,32,15,1),s=0;s<e.f;s++)((e.u[s]&3)==1||(e.u[s]&3)==2)&&(ot=true,wt=(e.u[s]&U1)>>18,wt==1?++i[(e.u[s]&U1)>>18!=1&&(e.u[s]&U1)>>18!=2?-1:(e.u[s]&k0)>>20]:wt==2?++Qt[(e.u[s]&U1)>>18!=1&&(e.u[s]&U1)>>18!=2?-1:(e.u[s]&k0)>>20]:++P);for(r=0,Dt=z(l.d7,H,6,32,15,1),D=z(l.d7,H,6,32,15,1),d=0;d<e.g;d++)n.em(e,d)!=2&&((e.F[d]&3)==1||(e.F[d]&3)==2)&&(ot=true,wt=(e.F[d]&768)>>8,wt==1?++D[(e.F[d]&768)>>8!=1&&(e.F[d]&768)>>8!=2?-1:(e.F[d]&S0)>>10]:wt==2?++Dt[(e.F[d]&768)>>8!=1&&(e.F[d]&768)>>8!=2?-1:(e.F[d]&S0)>>10]:++r);if(ot){if(T.Njb(t.b,"M  V30 BEGIN COLLECTION"+I),P!=0){for(T.Njb(t.b,"M  V30 MDLV30/STEABS ATOMS=("+P),u=0;u<e.f;u++)((e.u[u]&3)==1||(e.u[u]&3)==2)&&(e.u[u]&U1)>>18==0&&T.Njb(t.b," "+(u+1));T.Njb(t.b,")"+I);}if(r!=0){for(T.Njb(t.b,"M  V30 MDLV30/STEABS BONDS=("+r),c=0;c<e.g;c++)n.em(e,c)!=2&&((e.F[c]&3)==1||(e.F[c]&3)==2)&&(e.F[c]&768)>>8==0&&T.Njb(t.b," "+(c+1));T.Njb(t.b,")"+I);}for(M=0;M<32;M++){if(Qt[M]!=0){for(T.Njb(t.b,J8+(M+1)+" ATOMS=("+Qt[M]),u=0;u<e.f;u++)((e.u[u]&3)==1||(e.u[u]&3)==2)&&(e.u[u]&U1)>>18==2&&((e.u[u]&U1)>>18!=1&&(e.u[u]&U1)>>18!=2?-1:(e.u[u]&k0)>>20)==M&&T.Njb(t.b," "+(u+1));T.Njb(t.b,")"+I);}if(i[M]!=0){for(T.Njb(t.b,Z8+(M+1)+" ATOMS=("+i[M]),u=0;u<e.f;u++)((e.u[u]&3)==1||(e.u[u]&3)==2)&&(e.u[u]&U1)>>18==1&&((e.u[u]&U1)>>18!=1&&(e.u[u]&U1)>>18!=2?-1:(e.u[u]&k0)>>20)==M&&T.Njb(t.b," "+(u+1));T.Njb(t.b,")"+I);}if(Dt[M]!=0){for(T.Njb(t.b,J8+(M+1)+" BONDS=("+Dt[M]),c=0;c<e.g;c++)n.em(e,c)!=2&&((e.F[c]&3)==1||(e.F[c]&3)==2)&&(e.F[c]&768)>>8==2&&((e.F[c]&768)>>8!=1&&(e.F[c]&768)>>8!=2?-1:(e.F[c]&S0)>>10)==M&&T.Njb(t.b," "+(c+1));T.Njb(t.b,")"+I);}if(D[M]!=0){for(T.Njb(t.b,Z8+(M+1)+" BONDS=("+D[M]),c=0;c<e.g;c++)n.em(e,c)!=2&&((e.F[c]&3)==1||(e.F[c]&3)==2)&&(e.F[c]&768)>>8==1&&((e.F[c]&768)>>8!=1&&(e.F[c]&768)>>8!=2?-1:(e.F[c]&S0)>>10)==M&&T.Njb(t.b," "+(c+1));T.Njb(t.b,")"+I);}}T.Njb(t.b,"M  V30 END COLLECTION"+I);}T.Njb(t.b,Lr+I);},n.Hs=function(){n.Es(this),this.b=new T.Qjb();},n.Is=function(t){n.Js.call(this,t);},n.Js=function(t){n.Ks.call(this,t,new T.Qjb());},n.Ks=function(t,e){var A,P,r,i;n.Es(this),this.a=new o0.Nmb((P=rP,P)),n.aw(t,15),i=(T.Ujb(),p1),this.b=e,r=t.P!=null?t.P:"",T.Njb(this.b,r+i),T.Njb(this.b,"Actelion Java MolfileCreator 2.0"+i+i),T.Njb(this.b,"  0  0  0  0  0  0              0 V3000"+i),this.c=1,A=n.Ms(t),A&&(this.c=n.Ls(t)),n.Gs(this,t,A),T.Njb(this.b,me+i);},n.Ls=function(t){var e,A,P,r,i,D,u,o,Q;if(Q=1,P=n.Vl(t,t.q,t.r,(n.Qk(),n.Pk)),P!=0)(P<1||P>3)&&(Q=1.5/P);else {for(o=_8,e=1;e<t.q;e++)for(A=0;A<e;A++)i=t.J[A].a-t.J[e].a,D=t.J[A].b-t.J[e].b,u=t.J[A].c-t.J[e].c,r=i*i+D*D+u*u,o>r&&(o=r);Q=3/j.Math.max(.75,o);}return Q;},n.Ms=function(t){var e;if(t.q==1)return  true;for(e=1;e<t.q;e++)if(t.J[e].a!=t.J[0].a||t.J[e].b!=t.J[0].b||t.J[e].c!=t.J[0].c)return  true;return  false;},n.Ns=function(t,e){var A;return A=new n.Hs(),A.c=e,n.aw(t,15),n.Gs(A,t,true),A.b.a;},x(149,1,{},n.Hs,n.Is),y.c=1,l.P7=X(149),n.Qs=function(){n.Qs=o1,n.Os=F(k(l.Sbb,2),c0,13,0,[F(k(l.Sbb,1),st,2,6,["Acyl","gCaHA`AIf`@"]),F(k(l.Sbb,1),st,2,6,["Alloc","gNph@l@ILzuR@@"]),F(k(l.Sbb,1),st,2,6,["Allyl","Al"]),F(k(l.Sbb,1),st,2,6,["Benzyl;Bn","daD@`F@DjUZxHH@@"]),F(k(l.Sbb,1),st,2,6,["Boc","daxD`@S@AIgijj@@"]),F(k(l.Sbb,1),st,2,6,["BOM;BzOM","deTH`@H@Re[TYj`@@@"]),F(k(l.Sbb,1),st,2,6,["Bs","dmtDpAdLS`aPAIe]jf`@e`@@"]),F(k(l.Sbb,1),st,2,6,["Bt","dew@`@aJ@DiY]paej`@@@"]),F(k(l.Sbb,1),st,2,6,["Btm","did@P@BJ@Die_ahH@@@"]),F(k(l.Sbb,1),st,2,6,["Butyl;nButyl","gJPHB@IRuP@"]),F(k(l.Sbb,1),st,2,6,["Benzoyl;Bz","didH`@p@RYm^Eh@@@@"]),F(k(l.Sbb,1),st,2,6,["Bzh","dg|@`N@LdbRbtJUB]aAP@@@@"]),F(k(l.Sbb,1),st,2,6,["Cbz","dmtD`@S@AIgYVUZh@@@@"]),F(k(l.Sbb,1),st,2,6,["cButyl","gKPHL@IThuT@@"]),F(k(l.Sbb,1),st,2,6,["cHeptyl","daD@`L@DjWVzjj`@"]),F(k(l.Sbb,1),st,2,6,["cHexyl","gOpHL@IToWUU@@"]),F(k(l.Sbb,1),st,2,6,["cOctyl","did@`L@DjWWajjj@@"]),F(k(l.Sbb,1),st,2,6,["cPentyl","gFpHL@ITimUP@"]),F(k(l.Sbb,1),st,2,6,["cPropyl","gBPHL@Qxjh@"]),F(k(l.Sbb,1),st,2,6,["COOH","gC`h@l@ILt@@ !Bb@K~@Hc}"]),F(k(l.Sbb,1),st,2,6,["DEAE","daz@`@x@RiUjj`@"]),F(k(l.Sbb,1),st,2,6,["DEIPS","diD@P@\\B@DjfVjj`@"]),F(k(l.Sbb,1),st,2,6,["DMIPS","gNpD@xD@RjZjh@"]),F(k(l.Sbb,1),st,2,6,["DMPM","dcLD`@kPCIEMDdcttDDT@@"]),F(k(l.Sbb,1),st,2,6,["DMPS","deT@P@\\B@LddTjPsU@@@@"]),F(k(l.Sbb,1),st,2,6,["DMTr","fak@b@@Mt@ISZ{SMjo{NQKfm@AU@@@E@@@@"]),F(k(l.Sbb,1),st,2,6,["DNP","dkmB`hdDt~@HeNfS{HihheCAUhBHX@@"]),F(k(l.Sbb,1),st,2,6,["DNS;Dan","fhi`a@KPP@HH@YIHYheEhYKQgKP@@QP@@@"]),F(k(l.Sbb,1),st,2,6,["DPIPS","fdyAA@H@\\B@FRRIQSQIHzp_Qjh@h@@@@@"]),F(k(l.Sbb,1),st,2,6,["DPTBS","fleAA@H@\\B@FRRIQSRIIWNbEMU@EP@@@@@"]),F(k(l.Sbb,1),st,2,6,["DTBMS","dmT@P@\\B@Djffjjjh@@"]),F(k(l.Sbb,1),st,2,6,["Ethyl","eMBD@ch@"]),F(k(l.Sbb,1),st,2,6,["Fmoc","fde@b@@Hp@IL{LrjxeVCzKUT@@@P@@@"]),F(k(l.Sbb,1),st,2,6,["iAmyl","gGPHJ@YIDZj@@"]),F(k(l.Sbb,1),st,2,6,["iButyl","gJPHB@ITuP@"]),F(k(l.Sbb,1),st,2,6,["Im","gFtHAj@IRnKSP@"]),F(k(l.Sbb,1),st,2,6,["iPropyl","gC`HL@Qz`@"]),F(k(l.Sbb,1),st,2,6,["MDIPS","diD@P@\\B@DjfZjj`@"]),F(k(l.Sbb,1),st,2,6,["MDPS","foA@A@@NA@CIIEEBdeeVLzj@@@@@@"]),F(k(l.Sbb,1),st,2,6,["Methyl","eFBH@c@@"]),F(k(l.Sbb,1),st,2,6,["MEM","gNphAR@IRoUT@@"]),F(k(l.Sbb,1),st,2,6,["Mes","deT@`J@DjY{[`bB`@@"]),F(k(l.Sbb,1),st,2,6,["MMTr","ffcAB@@Z@Dim]ifuWYrI\\uh@Jh@@@@@@"]),F(k(l.Sbb,1),st,2,6,["MOM","gCaHA`AJZ`@"]),F(k(l.Sbb,1),st,2,6,["MPM;PMB","deTH`@d@Rfuunh@J@@"]),F(k(l.Sbb,1),st,2,6,["Ms","gJPdH`DD@cuh@"]),F(k(l.Sbb,1),st,2,6,["MTM","gC`D@DX@Rfh@"]),F(k(l.Sbb,1),st,2,6,["mTolyl","daD@`N@DjWjXHB@@"]),F(k(l.Sbb,1),st,2,6,["N3","gClHaE`@RnReX@"]),F(k(l.Sbb,1),st,2,6,["nAmyl;Amyl","gGPHJ@IRmU@@"]),F(k(l.Sbb,1),st,2,6,["neoAm","gGPHJ@IUMU@@"]),F(k(l.Sbb,1),st,2,6,["NO2,Nitro","gChhhE`BRnRYh@"]),F(k(l.Sbb,1),st,2,6,["Np","deVDaHAI@HeNR[e_aZ@B@@"]),F(k(l.Sbb,1),st,2,6,["nPropyl;Propyl","gC`HL@IST@@"]),F(k(l.Sbb,1),st,2,6,["oTolyl","daD@`J@DjYvxH`@@"]),F(k(l.Sbb,1),st,2,6,["Phenyl","gOpHL@IToVD@@@"]),F(k(l.Sbb,1),st,2,6,["Pht","dcLL`@RU@Dfyed]ZBA`@@"]),F(k(l.Sbb,1),st,2,6,["Piv;Pv","gNqHA`AIffj`@"]),F(k(l.Sbb,1),st,2,6,["PMBM","dcLD`@T`AJUm]FZh@J@@"]),F(k(l.Sbb,1),st,2,6,["PNB","dcNLaHAEt@bTyInUvxV`@f@@"]),F(k(l.Sbb,1),st,2,6,["Poc","didD`@S@AIgexVjj`@"]),F(k(l.Sbb,1),st,2,6,["PPi","diDFsHSB[`|J|A@Lxn{lddqdZih@@"]),F(k(l.Sbb,1),st,2,6,["pTolyl","daD@`N@DjWzXHB@@"]),F(k(l.Sbb,1),st,2,6,["sAmyl","gGPHL@YIDZj@@"]),F(k(l.Sbb,1),st,2,6,["sButyl","gJPHL@ITuP@"]),F(k(l.Sbb,1),st,2,6,["SEM","diDHPFApD@rRQUJjj`@"]),F(k(l.Sbb,1),st,2,6,["SES","dedDpHP@``AgCIICeHmUT@@"]),F(k(l.Sbb,1),st,2,6,["SO3H","gJQdHl@``D^m@@"]),F(k(l.Sbb,1),st,2,6,["tAmyl","gGPHB@IUMU@@"]),F(k(l.Sbb,1),st,2,6,["TBDMS;TBS","dax@P@\\B@Djfjjh@@"]),F(k(l.Sbb,1),st,2,6,["TBDPS","fdy@A@@NA@CIIEEEIde]XOhuPAT@@@@@"]),F(k(l.Sbb,1),st,2,6,["TBMPS","dg\\HPHApH@rRQJJPjg]UAT@@@"]),F(k(l.Sbb,1),st,2,6,["tButyl,tBu","gJPHB@Q}T@@"]),F(k(l.Sbb,1),st,2,6,["TDS","ded@P@\\B@LddTeeUUP@@"]),F(k(l.Sbb,1),st,2,6,["Tf","daxDhHP@``BiAiCiCIICHmU@@"]),F(k(l.Sbb,1),st,2,6,["TFA","gNqBJIARFdF@YEHYUL@@"]),F(k(l.Sbb,1),st,2,6,["Thexyl","gNpHB@IUMUT@@"]),F(k(l.Sbb,1),st,2,6,["THF","gFqH@PAJYujj@@"]),F(k(l.Sbb,1),st,2,6,["THP","gOqH@PAJYZzjh@"]),F(k(l.Sbb,1),st,2,6,["TIPS","dmT@P@\\B@DjfYjjjh@@"]),F(k(l.Sbb,1),st,2,6,["TMS","gJPD@xD@czh@"]),F(k(l.Sbb,1),st,2,6,["Tosyl;Ts","dmtDPHP@``CIICLeaeZ@B@@"]),F(k(l.Sbb,1),st,2,6,["Troc","diDDHJxHaHcH`PCHiBeJjf@@"]),F(k(l.Sbb,1),st,2,6,["Trt","fbm@B@A@FRQIRKQPiIZdoIcdHJ`@@@@@@"]),F(k(l.Sbb,1),st,2,6,["Xyl","did@`J@DjYynBHH@@"])]);},n.Rs=function(){n.Qs();var t,e,A,P,r,i,D,u,o;for(n.Ps=new U.awb(),D=n.Os,u=0,o=D.length;u<o;++u)for(i=D[u],r=T.sjb(i[0],";"),e=r,A=0,P=e.length;A<P;++A)t=e[A],U.Uvb(n.Ps,T.qjb(w.Xzb(t).toLowerCase(),"-"),i[1]);},n.Ss=function(t){n.Qs();var e,A,P,r,i,D,u,o,Q,L;if(t==null||w.Xzb(t).length==0)return null;for(D=null,t=T.qjb(w.Xzb(t).toLowerCase(),"-"),o=n.Os,Q=0,L=o.length;Q<L;++Q)for(u=o[Q],i=T.sjb(u[0],";"),A=i,P=0,r=A.length;P<r;++P){if(e=A[P],T.ejb(T.qjb(w.Xzb(e).toLowerCase(),"-"),t))return e;if(T.tjb(T.qjb(w.Xzb(e).toLowerCase(),"-"),t))if(D==null)D=e;else return "";}return D;},n.Ps=null,n.Vs=function(){var t,e,A,P;for(this.b=new U.utb(),this.c=new U.utb(),this.d=new U.utb(),n.Ts=F(k(l.B7,1),C1,18,0,[new n.lk(1,"Hydrogen","H",2.2),new n.lk(2,"Helium","He",0),new n.lk(3,"Lithium","Li",.98),new n.lk(4,"Beryllium","Be",1.57),new n.lk(5,"Boron","B",2.04),new n.lk(6,"Carbon","C",2.55),new n.lk(7,"Nitrogen","N",3.04),new n.lk(8,"Oxygen","O",3.44),new n.lk(9,"Fluorine","F",3.98),new n.lk(10,"Neon","Ne",0),new n.lk(11,"Sodium","Na",.93),new n.lk(12,"Magnesium","Mg",1.31),new n.lk(13,"Aluminum","Al",1.61),new n.lk(14,"Silicon","Si",1.9),new n.lk(15,"Phosphorus","P",2.19),new n.lk(16,"Sulfur","S",2.58),new n.lk(17,"Chlorine","Cl",3.16),new n.lk(18,"Argon","Ar",0),new n.lk(19,"Potassium","K",1),new n.lk(20,"Calcium","Ca",1.36),new n.lk(21,"Scandium","Sc",1.54),new n.lk(22,"Titanium","Ti",1.63),new n.lk(23,"Vanadium","V",1.66),new n.lk(24,"Chromium","Cr",1.55),new n.lk(25,"Manganese","Mn",1.83),new n.lk(26,"Iron","Fe",1.88),new n.lk(27,"Cobalt","Co",1.91),new n.lk(28,"Nickel","Ni",1.9),new n.lk(29,"Copper","Cu",1.65),new n.lk(30,"Zinc","Zn",1.81),new n.lk(31,"Gallium","Ga",2.01),new n.lk(32,"Germanium","Ge",0),new n.lk(33,"Arsenic","As",2.18),new n.lk(34,"Selenium","Se",2.55),new n.lk(35,"Bromine","Br",0),new n.lk(36,"Krypton","Kr",2.96),new n.lk(37,"Rubidium","Rb",0),new n.lk(38,"Strontium","Sr",.82),new n.lk(39,"Yttrium","Y",.95),new n.lk(40,"Zirconium","Zr",1.22),new n.lk(41,"Niobium","Nb",1.33),new n.lk(42,"Molybdenum","Mo",1.6),new n.lk(43,"Technetium","Tc",2.16),new n.lk(44,"Ruthenium","Ru",1.9),new n.lk(45,"Rhodium","Rh",2.2),new n.lk(46,"Palladium","Pd",2.28),new n.lk(47,"Silver","Ag",2.2),new n.lk(48,"Cadmium","Cd",1.93),new n.lk(49,"Indium","In",1.69),new n.lk(50,"Tin","Sn",1.78),new n.lk(51,"Antimony","Sb",1.96),new n.lk(52,"Tellurium","Te",0),new n.lk(53,"Iodine","I",2.05),new n.lk(54,"Xenon","Xe",2.1),new n.lk(55,"Cesium","Cs",2.66),new n.lk(56,"Barium","Ba",2.6),new n.lk(57,"Lanthanum","La",.79),new n.lk(58,"Cerium","Ce",.89),new n.lk(59,"Praesodymium","Pr",1.1),new n.lk(60,"Neodymium","Nd",1.12),new n.lk(61,"Promethium","Pm",1.13),new n.lk(62,"Samarium","Sm",1.14),new n.lk(63,"Europium","Eu",0),new n.lk(64,"Gadolinium","Gd",1.17),new n.lk(65,"Terbium","Tb",0),new n.lk(66,"Dyprosium","Dy",1.2),new n.lk(67,"Holmium","Ho",0),new n.lk(68,"Erbium","Er",1.22),new n.lk(69,"Thulium","Tm",1.23),new n.lk(70,"Ytterbium","Yb",1.24),new n.lk(71,"Lutetium","Lu",1.25),new n.lk(72,"Hafnium","Hf",0),new n.lk(73,"Tantalium","Ta",1.27),new n.lk(74,"Wolfram","W",1.3),new n.lk(75,"Rhenium","Re",1.5),new n.lk(76,"Osmium","Os",2.36),new n.lk(77,"Iridium","Ir",1.9),new n.lk(78,"Platinum","Pt",2.2),new n.lk(79,"Gold","Au",2.2),new n.lk(80,"Mercury","Hg",2.28),new n.lk(81,"Thallium","Tl",2.54),new n.lk(82,"Lead","Pb",2),new n.lk(83,"Bismuth","Bi",1.62),new n.lk(84,"Polonium","Po",2.33),new n.lk(85,"Astatine","At",2.02),new n.lk(86,"Radon","Rn",2),new n.lk(87,"Francium","Fr",2.2),new n.lk(88,"Radium","Ra",0),new n.lk(89,"Actinium","Ac",.7),new n.lk(90,"Thorium","Th",.89),new n.lk(91,"Protactinium","Pa",1.1),new n.lk(92,"Uranium","U",1.3),new n.lk(93,"Neptunium","Np",1.5),new n.lk(94,"Plutonium","Pu",1.38),new n.lk(95,"Americium","Am",1.36),new n.lk(96,"Curium","Cm",1.28),new n.lk(97,"Berkelium","Bk",1.3),new n.lk(98,"Californium","Cf",1.3),new n.lk(99,"Einsteinium","E",1.3),new n.lk(100,"Fermium","Fm",1.3),new n.lk(101,"Mendelevium","Mv",1.3)]),e=n.Ts,A=0,P=e.length;A<P;++A)t=e[A],U.vnb(this.b,T.uib(t.c),t),U.wnb(this.c,t.b,t),U.wnb(this.d,w.Xzb(t.d).toLowerCase(),t);this.a=z(l.zeb,ht,6,n.Ts.length,16,1),this.a[3]=true,this.a[11]=true,this.a[19]=true,this.a[37]=true,this.a[55]=true,this.a[87]=true;},n.Ws=function(t){return U.snb((!n.Us&&(n.Us=new n.Vs()),n.Us).b,T.uib(t));},x(393,1,{},n.Vs),y.ob=function(){return "";},l.Q7=X(393),n.Ys=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g;for(Q=t.i.M,L=0,i=0;i<A;i++)Q>e[i]&&(Q=e[i],L=i);for(g=z(l.d7,H,6,A,15,1),o=L>0?L-1:A-1,s=L<A-1?L+1:0,D=e[o]<e[s],r=0;r<A;r++)g[r]=e[L],D?--L<0&&(L=A-1):++L==A&&(L=0);for(c=new U.wqb(t.j);c.a<c.c.a.length;)if(b=U.vqb(c),b.length==A){for(P=true,u=0;u<A;u++)if(b[u]!=g[u]){P=false;break;}if(P)return;}U.rj(t.j,g),h=n.it(t,g),U.rj(t.k,h),n.At(t,g,h);},n.Zs=function(t,e,A){var P,r,i,D,u,o,Q;for(o=z(l.d7,H,6,t.g,15,1),D=z(l.d7,H,6,t.g,15,1),Q=z(l.zeb,ht,6,t.i.f,16,1),P=n.Yl(t.i,0,e),r=n.Yl(t.i,1,e),o[0]=P,o[1]=r,D[1]=-1,Q[r]=true,u=1;u>=1;){if(++D[u],D[u]==n.Zo(t.i,o[u])){Q[o[u]]=false,--u;continue;}if(i=n.Yo(t.i,o[u],D[u]),!(Q[i]||A[i])){if(i==P&&u>1){if(n.Ys(t,o,u+1),t.j.a.length>=i0)return;continue;}u+1<t.g&&(++u,o[u]=i,Q[i]=true,D[u]=-1);}}},n.$s=function(t,e,A,P,r,i,D,u){var o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt;for(Lt=U.wj(t.j,e),s=Lt,h=0,b=s.length;h<b;++h)if(L=s[h],!n.wt(t,L))return  true;for(ot=U.wj(t.k,e),rt=ot.length,g=0,Q=0,wt=false,O=0;O<rt;O++)g<<=1,Q<<=1,n.yt(t,ot[O])?g|=1:u&&n.vt(t,ot[O])?(g|=1,Q|=1):(o=A[e][O],o!=-1&&(P[o]?r[o]&&(g|=1,i[o]||(Q|=1)):wt=true));switch(S=false,rt){case 3:for(d=F(k(l.d7,1),H,6,15,[2,1,4]),S=true,M=0;M<3;M++)(g&d[M])==d[M]&&(n.Sl(t.i,Lt[M])==6&&n.Al(t.i,Lt[M])==1||n.Sl(t.i,Lt[M])==5&&n.Al(t.i,Lt[M])==0)&&(r[e]=true,D[e]=M,(Q&d[M])==0&&(S=false));break;case 5:for(m=F(k(l.d7,1),H,6,15,[10,5,18,9,20]),S=true,Dt=0;Dt<5;Dt++)if((g&m[Dt])==m[Dt])switch(n.Sl(t.i,Lt[Dt])){case 6:n.Al(t.i,Lt[Dt])==-1&&(r[e]=true,D[e]=Dt,(Q&m[Dt])==0&&(S=false));break;case 7:n.Al(t.i,Lt[Dt])<=0&&(r[e]=true,D[e]=Dt);break;case 8:r[e]=true,D[e]=Dt;break;case 16:case 34:case 52:n.Zo(t.i,Lt[Dt])==2&&(r[e]=true,D[e]=Dt);}break;case 6:S=true,(g&21)==21&&(r[e]=true,(Q&21)==0&&(S=false)),(g&42)==42&&(r[e]=true,(Q&42)==0&&(S=false));break;case 7:for(C=F(k(l.d7,1),H,6,15,[42,21,74,37,82,41,84]),S=true,v=0;v<7;v++)if((g&C[v])==C[v]){if(n.Sl(t.i,Lt[v])==6&&(n.Al(t.i,Lt[v])==1||u&&n.lt(t,Lt[v]))||n.Sl(t.i,Lt[v])==5&&n.Al(t.i,Lt[v])==0)r[e]=true,D[e]=v,(Q&C[v])==0&&(S=false);else for(V=0;V<2;V++)if(bt=(v+6+V)%7,yt=ot[bt],it=A[e][bt],it!=-1&&!r[it]&&U.wj(t.k,it).length==5){for(ct=U.wj(t.k,it),K=0;ct[K]!=yt;)++K;for(N=false,Qt=0,I=1;I<=4;I++){if(c=ct[(K+I)%5],ut=n.em(t.i,c)>1||n.hm(t.i,c)==8||u&&n.vt(t,c),ut&&N){Qt=0;break;}N=ut,N&&++Qt;}if(Qt==2){r[e]=true,r[it]=true,i[e]=true,i[it]=true,U.rj(t.b,T.uib(yt));break;}}}}return r[e]&&!S&&(i[e]=true),r[e]?true:!wt;},n._s=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c,g,d;for(i=z(l.d7,S1,7,t.j.a.length,0,2),Q=0;Q<t.j.a.length;Q++)for(i[Q]=z(l.d7,H,6,U.wj(t.j,Q).length,15,1),L=0;L<U.wj(t.j,Q).length;L++)i[Q][L]=-1;for(g=z(l.d7,H,6,t.i.g,15,1),b=0;b<t.k.a.length;b++)if(c=U.wj(t.k,b),c.length==3||c.length>=5&&c.length<=7)for(o=0;o<c.length;o++)u=c[o],n.Zo(t.i,n.Yl(t.i,0,u))==3&&n.Zo(t.i,n.Yl(t.i,1,u))==3&&(g[u]>0?(i[g[u]>>>16][g[u]&32767]=b,i[b][o]=g[u]>>>16):g[u]=(b<<16)+Oe+o);for(D=z(l.zeb,ht,6,t.j.a.length,16,1),d=0,s=-1;d>s;)for(s=d,h=0;h<t.j.a.length;h++)D[h]||n.$s(t,h,i,D,e,A,P,r)&&(D[h]=true,++d);},n.at=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;for(r=n.Yl(t.i,0,e),i=n.Yl(t.i,1,e),o=z(l.d7,H,6,t.i.f,15,1),Q=z(l.d7,H,6,t.i.f,15,1),L=z(l.d7,H,6,t.i.f,15,1),o[0]=r,o[1]=i,Q[r]=1,Q[i]=2,L[r]=-1,L[i]=r,u=1,s=1;u<=s;){for(h=0;h<n.Zo(t.i,o[u]);h++){if(D=n.Yo(t.i,o[u],h),u>1&&D==r){for(c=z(l.d7,H,6,Q[o[u]],15,1),P=o[u],b=0;b<c.length;b++)c[b]=P,P=L[P];return c;}Q[D]==0&&!A[D]&&(o[++s]=D,Q[D]=Q[o[u]]+1,L[D]=o[u]);}++u;}return null;},n.bt=function(t,e,A){var P,r;for(r=U.wj(t.j,e),P=0;P<r.length;P++)if(A==r[P])return P;return  -1;},n.ct=function(t,e){return t.a[e]&r1;},n.dt=function(t,e,A){var P,r;for(r=U.wj(t.k,e),P=0;P<r.length;P++)if(A==r[P])return P;return  -1;},n.et=function(t,e){return t.c[e]&r1;},n.ft=function(t,e){return t.d[e];},n.gt=function(t,e){return U.wj(t.j,e);},n.ht=function(t,e){return U.wj(t.k,e);},n.it=function(t,e){var A,P,r,i,D;for(i=e.length,D=z(l.d7,H,6,i,15,1),P=0;P<i;P++)for(A=P==i-1?e[0]:e[P+1],r=0;r<n.Zo(t.i,e[P]);r++)if(n.Yo(t.i,e[P],r)==A){D[P]=n.$o(t.i,e[P],r);break;}return D;},n.jt=function(t,e){return U.wj(t.k,e).length;},n.kt=function(t,e,A){var P;for(P=0;P<t.k.a.length;P++)if(n.qt(t,P,e)&&n.qt(t,P,A))return P;return  -1;},n.lt=function(t,e){var A;for(A=0;A<n.Zo(t.i,e);A++)if(n._o(t.i,e,A)==2&&n.Sl(t.i,n.Yo(t.i,e,A))==8)return  true;return  false;},n.mt=function(t,e){return t.e[e];},n.nt=function(t,e){return (t.a[e]&q1)!=0;},n.ot=function(t,e){return (t.c[e]&q1)!=0;},n.pt=function(t,e,A){var P,r,i,D;for(r=U.wj(t.j,e),i=0,D=r.length;i<D;++i)if(P=r[i],A==P)return  true;return  false;},n.qt=function(t,e,A){var P,r,i,D;for(r=U.wj(t.k,e),i=0,D=r.length;i<D;++i)if(P=r[i],A==P)return  true;return  false;},n.rt=function(t,e){return t.f[e];},n.st=function(t,e){return (t.a[e]&A0)!=0;},n.tt=function(t,e){return (t.c[e]&A0)!=0;},n.ut=function(t,e){return (t.a[e]&Ue)!=0;},n.vt=function(t,e){var A,P,r,i,D,u;for(D=0;D<2;D++)if(A=n.Yl(t.i,D,e),n.Sl(t.i,A)==7&&n.Zo(t.i,A)==2&&(P=n.Yl(t.i,1-D,e),n.Sl(t.i,P)==6)){for(u=0;u<n.Zo(t.i,P);u++)if(r=n.Yo(t.i,P,u),i=n.$o(t.i,P,u),(n.Sl(t.i,r)==8||n.Sl(t.i,r)==16)&&n.em(t.i,i)==2&&n.Zo(t.i,r)==1)return  true;}return  false;},n.wt=function(t,e){var A,P,r,i,D;if(t.i.K){if(kt(G(n.Nl(t.i,e),1),0))return  true;if(D=n.Il(t.i,e),D!=null){for(P=D,r=0,i=P.length;r<i;++r)if(A=P[r],A==5||A==6||A==7||A==8||A==15||A==16||A==33||A==34)return  true;return  false;}}return n.Et(n.Sl(t.i,e));},n.xt=function(t,e){var A,P,r,i,D;if(t.i.K){if(kt(G(n.Nl(t.i,e),1),0))return  false;if(A=n.Il(t.i,e),A!=null){for(r=A,i=0,D=r.length;i<D;++i)if(P=r[i],!n.po(P))return  false;return  true;}}return n.po(n.Sl(t.i,e));},n.yt=function(t,e){return n.em(t.i,e)>1||n.hm(t.i,e)==8;},n.zt=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;for(m=0;m<t.e.length;m++)if(t.e[m]){for(d=false,P=U.wj(t.j,m),i=0,u=P.length;i<u;++i)e=P[i],t.a[e]|=q1,n.xt(t,e)&&(d=true);for(L=U.wj(t.k,m),h=0,g=L.length;h<g;++h)o=L[h],t.c[o]|=q1;if(t.f[m]){for(A=U.wj(t.j,m),r=0,D=A.length;r<D;++r)e=A[r],t.a[e]|=A0;for(Q=U.wj(t.k,m),s=0,c=Q.length;s<c;++s)o=Q[s],t.c[o]|=A0;}if(d){for(A=U.wj(t.j,m),r=0,D=A.length;r<D;++r)e=A[r],t.a[e]|=Ue;for(Q=U.wj(t.k,m),s=0,c=Q.length;s<c;++s)o=Q[s],t.c[o]|=Ue;}}for(b=new U.wqb(t.b);b.a<b.c.a.length;)o=U.vqb(b).a,t.c[o]=0;},n.At=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h;for(h=e.length,r=e,i=0,D=r.length;i<D;++i)P=r[i],s=t.a[P]&r1,(s==0||s>h)&&(t.a[P]&=fe,t.a[P]|=h);for(o=A,Q=0,L=o.length;Q<L;++Q)u=o[Q],s=t.c[u]&r1,(s==0||s>h)&&(t.c[u]&=fe,t.c[u]|=h);},n.Bt=function(t,e,A){var P;for(P=U.wj(t.k,e).length;A>=P;)A-=P;for(;A<0;)A+=P;return A;},n.Ct=function(t,e){n.Dt.call(this,t,e);},n.Dt=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;this.i=t,this.g=7,this.j=new U.Lj(),this.k=new U.Lj(),this.b=new U.Lj(),this.a=z(l.d7,H,6,this.i.f,15,1),this.c=z(l.d7,H,6,this.i.g,15,1),this.i.Zb(1),h=z(l.zeb,ht,6,this.i.f,16,1),b=z(l.zeb,ht,6,this.i.g,16,1);do for(D=false,A=0;A<this.i.f;A++)if(!h[A]){for(d=0,s=0;s<n.Zo(this.i,A);s++)h[n.Yo(this.i,A,s)]||++d;if(d<2){for(h[A]=true,L=0;L<n.Zo(this.i,A);L++)b[n.$o(this.i,A,L)]=true;D=true;}}while(D);for(C=0;C<this.i.f&&h[C];)++C;if(C!=this.i.f){for(o=z(l.d7,H,6,this.i.f,15,1),o[0]=C,g=z(l.d7,H,6,this.i.f,15,1),g[0]=-1,u=z(l.d7,H,6,this.i.f,15,1),u[C]=1,i=0,Q=0,c=1;i<=Q;){for(L=0;L<n.Zo(this.i,o[i]);L++)if(r=n.Yo(this.i,o[i],L),r!=g[o[i]]){if(u[r]!=0){n.Zs(this,n.$o(this.i,o[i],L),h);continue;}h[r]||(u[r]=c,g[r]=o[i],o[++Q]=r);}if(++i,i>Q){for(A=0;A<this.i.f;A++)if(u[A]==0&&!h[A]){u[A]=++c,o[++Q]=A,g[A]=-1;break;}}}if((e&4)!=0&&(this.e=z(l.zeb,ht,6,this.j.a.length,16,1),this.f=z(l.zeb,ht,6,this.j.a.length,16,1),this.d=z(l.d7,H,6,this.j.a.length,15,1),n._s(this,this.e,this.f,this.d,(e&8)!=0),n.zt(this)),(e&2)!=0)for(P=0;P<this.i.g;P++)!b[P]&&n.em(this.i,P)!=0&&(m=n.at(this,P,h),m!=null&&n.At(this,m,n.it(this,m)));}},n.Et=function(t){return t==5||t==6||t==7||t==8||t==15||t==16||t==33||t==34;},x(150,1,{},n.Ct),y.g=0,l.R7=X(150),n.Ft=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;if(e==2||e==5)U.rj(t.F,n.ju(t.G,t.G.length)),t.b&&U.rj(t.a,n.ku(t.C,t.C.length));else if(e==4)h=n.Rt(n.ju(t.G,t.G.length)),U.Pwb(t.O,h)||(U.Nwb(t.O,h),U.rj(t.F,n.ju(t.G,t.G.length)),t.b&&U.rj(t.a,n.ku(t.C,t.C.length)));else if(e==3){if(h=n.Rt(n.ju(t.G,t.G.length)),!U.Pwb(t.O,h)){for(s=false,Q=(u=new U.kwb(new U.qwb(new U.xpb(t.O.a).a).b),new U.Gpb(u));U.fob(Q.a.a);){for(o=(D=U.iwb(Q.a),D.fd()),L=0,P=h,r=0,i=P.length;r<i;++r){for(A=P[r];L<o.length&&o[L]<A;)++L;if(L<o.length&&A==o[L]){s=true;break;}}if(s)break;}s||(U.Nwb(t.O,h),U.rj(t.F,n.ju(t.G,t.G.length)),t.b&&U.rj(t.a,n.ku(t.C,t.C.length)));}}else e==6&&(h=n.St(t,n.ju(t.G,t.G.length)),U.Pwb(t.O,h)||(U.Nwb(t.O,h),U.rj(t.F,n.ju(t.G,t.G.length)),t.b&&U.rj(t.a,n.ku(t.C,t.C.length))));},n.Gt=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b;if(u=n.Zo(t.H,e),P=t.o[A],P>u)return  false;if(Q=n.Nl(t.H,e),i=n.Nl(t.g,A),r=n.Il(t.g,A),o=n.Il(t.H,e),kt(G(i,1),0)){if(r!=null){if(kt(G(Q,1),0)){if(o==null||!n.Xt(r,o))return  false;}else if(o!=null){if(n.Zt(o,r))return  false;}else if(n.Wt(n.Sl(t.H,e),r))return  false;}}else {if(kt(G(Q,1),0))return  false;if(r!=null){if(o!=null){if(!n.Xt(o,r))return  false;}else if(!n.Wt(n.Sl(t.H,e),r))return  false;}else if(o!=null||t.J[e]!=t.k[A])return  false;}if(kt(Pt(Q,i),0)){if(kt(G(i,z1),0)){if(t.H.K&&M1(G(Q,z1),0))return  false;if(P!=u)return  false;}if(kt(G(i,Y1),0)&&P>=u&&M1(G(Q,Y1),0))return  false;}if(kt(G(t.I[e],te(t.j[A])),0)||kt(G(t.B[A],te(t.M[e])),0))return  false;if(D=G(i,Me),t.H.K){if(L=G(i,Me),et(L,0)!=0&&(et(D,0)==0||kt(G(D,te(L)),0)))return  false;}else if(et(D,0)!=0&&M1(G(D,t.M[e]),0))return  false;return n.Al(t.g,A)!=0&&n.Al(t.g,A)!=n.Al(t.H,e)||n.Ll(t.g,A)!=0&&n.Ll(t.g,A)!=n.Ll(t.H,e)||n.Ol(t.g,A)!=0&&n.Ol(t.g,A)!=n.Ol(t.H,e)||(s=G(i,ur),et(s,0)!=0&&n.wp(t.H,e)!=Gt(t0(s,47))-7)?false:(h=G(i,Be),et(h,0)!=0?(h=t0(h,22),t.H.K?(b=Gt(t0(G(n.Nl(t.H,e),Be),22)),Gt(h)==b):n.To(t.H,e)==Gt(h)):true);},n.Ht=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h;if(Q=t.K[e],D=t.n[A],(n.gm(t.g,A)&yn)!=0){if(o=n.im(t.H,e),r=n.im(t.g,A),i=n.gm(t.g,A)&127,o!=r&&!(o==1&&(i&1)!=0)&&!(o==2&&(i&2)!=0)&&!(o==4&&(i&4)!=0)&&!(o==32&&(i&32)!=0)&&!(o==64&&(i&64)!=0)&&!(o==16&&(i&16)!=0)&&!(o==8&&(i&8)!=0))return  false;Q&=-128,D&=-128;}if((Q&~D)!=0)return  false;if(h=(n.gm(t.g,A)&ie)>>17,h!=0){if(t.H.K&&h==(n.gm(t.H,A)&ie)>>17)return  true;if(h<=2)return L=n.Xo(t.H,e),h==1?L>=8&&L<=12:L>=12;for(P=false,s=n.Dp(t.H),u=0;u<s.j.a.length;u++)if(U.wj(s.k,u).length==h&&n.qt(s,u,e)){P=true;break;}if(!P)return  false;}return  true;},n.It=function(t){var e,A;for(t.b=null,e=0;e<t.g.g;e++)n.Bm(t.g,e)&&(!t.b&&(t.b=new U.Lj()),A=new n.lu(),A.a=n.Yl(t.g,0,e),A.b=n.Yl(t.g,1,e),A.d=n.$l(t.g,e),A.c=n.Zl(t.g,e),U.rj(t.b,A));},n.Jt=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h;for(n.aw(t.g,t.N),o=j.Math.max(t.g.f,t.g.g)+16,t.s=z(l.d7,H,6,o,15,1),t.u=z(l.d7,H,6,o,15,1),t.v=z(l.d7,H,6,o,15,1),t.t=z(l.zeb,ht,6,o+1,16,1),D=z(l.zeb,ht,6,t.g.f,16,1),u=z(l.zeb,ht,6,t.g.g,16,1),P=0,A=0;A<t.g.f;A++)if(!t.D[A]&&!D[A])for(t.s[P]=A,t.v[P]=-1,t.u[P]=-1,Q=P;P<=Q;){for(L=0;L<n.Lo(t.g,t.s[P]);L++)Q=n.gu(t,P,Q,L,D,u,-1);for(;t.t[++P];);}if(t.w=P,t.p!=0){for(Q=t.w-1,i=0;i<t.d;i++)for(P=0;P<=Q;){for(s=0;s<n.Lo(t.g,t.s[P]);s++)Q=n.gu(t,P,Q,s,D,u,i);for(;t.t[++P];);}for(e=0;e<t.g.f;e++)if(t.D[e]&&!D[e])for(t.s[P]=e,t.v[P]=-1,t.u[P]=-1,Q=P;P<=Q;){for(s=0;s<n.Lo(t.g,t.s[P]);s++)n.Yo(t.g,t.s[P],s)<t.g.f&&(Q=n.gu(t,P,Q,s,D,u,t.f[e]));for(;t.t[++P];);}for(t.e=z(l.d7,H,6,t.d,15,1),h=0;h<t.d;h++)t.e[h]=-1;for(L=t.w;L<P;L++)r=t.f[t.s[L]],t.e[r]==-1&&(t.e[r]=L);}t.A=P;},n.Kt=function(t,e){var A,P,r,i,D,u,o,Q;for(Q=n.Dp(t.H),u=0;u<Q.j.a.length;u++)if(n.qt(Q,u,e)){for(o=z(l.d7,H,6,2,15,1),r=0;r<2;r++)for(o[r]=St,P=n.Yl(t.H,r,e),i=0;i<n.Zo(t.H,P);i++)A=n.Yo(t.H,P,i),A!=n.Yl(t.H,1-r,e)&&o[r]>A&&(o[r]=A);return D=0,n.pt(Q,u,o[0])&&++D,n.pt(Q,u,o[1])&&++D,D==2?2:D==1?1:2;}return 0;},n.Lt=function(t,e,A){var P,r,i,D,u;if(t.b){for(t.C=z(l.zeb,ht,6,e.length,16,1),r=new U.wqb(t.b);r.a<r.c.a.length;)if(P=U.vqb(r),t.f==null||A==-1&&t.f[P.a]==-1&&t.f[P.b]==-1||A!=-1&&(t.f[P.a]==A||t.f[P.b]==A)){if(u=z(l.d7,H,6,P.c+2,15,1),i=n.xp(t.H,u,t.G[P.a],t.G[P.b],P.c+1,e,null)-1,i<P.d||i>P.c)return  false;for(D=1;D<=i;D++)t.C[u[D]]=true;}}return  true;},n.Mt=function(t,e){var A,P,r,i,D,u,o,Q;for(r=0;r<t.g.g;r++)if((n.gm(t.g,r)&se)!=0){if(i=n.fm(t.g,r),i==0)continue;if(A=n.Yl(t.g,0,r),P=n.Yl(t.g,1,r),t.f==null||e==-1&&t.f[A]==-1&&t.f[P]==-1||e!=-1&&(t.f[A]==e||t.f[P]==e)){if(D=t.G[A],u=t.G[P],o=n.Wo(t.H,D,u),Q=n.fm(t.H,o),Q==0&&(n.iq(t.H,o)&&(Q=n.Kt(t,o)),Q==0)||i==3||Q==3)continue;if(n.Tt(t,r)==(i==Q))return  false;}}return  true;},n.Nt=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g;for(D=0,o=0;o<t.g.f;o++)if((t.f==null||t.f[o]==e)&&kt(G(n.Nl(t.g,o),Q0),0)){if(h=t.G[o],s=n.Ml(t.g,o),c=n.Ml(t.H,h),s==0||s==3)continue;if(c==0||c==3)return  false;if(n.Gl(t.g,o)==1){++D;continue;}if(n.Gl(t.H,h)==1)return  false;if(n.Gl(t.g,o)==2){++D;continue;}if(n.Gl(t.H,h)==2||n.Yt(t,o)==(s==c))return  false;}if(D!=0){for(r=z(l.d7,H,6,D,15,1),i=0,Q=0;Q<t.g.f;Q++)(t.f==null||t.f[Q]==e)&&kt(G(n.Nl(t.g,Q),Q0),0)&&(s=n.Ml(t.g,Q),s!=0&&s!=3&&(r[i++]=n.Fl(t.g,Q)<<24|n.Gl(t.g,Q)<<22|Q));for(w.qzb(r,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),i=0;i<r.length;)for(L=r[i]&W1,b=t.G[L],A=r[i]&-4194304,P=n.Yt(t,L)^n.Ml(t.g,L)==n.Ml(t.H,b),++i;i<r.length&&(r[i]&-4194304)==A;i++)if(u=r[i]&W1,h=t.G[u],n.Gl(t.H,h)!=n.Gl(t.H,b)||n.Fl(t.H,h)!=n.Fl(t.H,b)||(g=n.Yt(t,u)^n.Ml(t.g,u)==n.Ml(t.H,h),g!=P))return  false;}return  true;},n.Ot=function(t,e,A){var P,r,i,D,u,o,Q;if(t.P=false,t.F=new U.Lj(),t.a=new U.Lj(),U.Nvb(t.O.a),!t.H||!t.g||t.g.f-t.p>t.H.f||t.g.g-t.q>t.H.g||t.g.f==0)return 0;for(e==6&&(t.N=63),n.cu(t,A),P=z(l.zeb,ht,6,t.H.f,16,1),t.G=z(l.d7,H,6,t.g.f,15,1),U.Iqb(t.G,-1),u=z(l.d7,H,6,t.A,15,1),U.Oqb(u,u.length,-1),i=0;!t.P;){if(t.w!=0){if(Q=t.u[i]==-1?t.H.f:n.Lo(t.H,t.G[t.u[i]]),++u[i],u[i]==Q){if(u[i]=-1,i==0)break;--i,t.t[i]||(P[t.G[t.s[i]]]=false);continue;}if(t.u[i]==-1)P[u[i]]||n.Gt(t,u[i],t.s[i])&&(t.G[t.s[i]]=u[i],P[u[i]]=true,++i);else {if(n.Yo(t.H,t.G[t.u[i]],u[i])>=t.H.f)continue;r=n.Yo(t.H,t.G[t.u[i]],u[i]),t.t[i]?r==t.G[t.s[i]]&&n.Ht(t,n.$o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&++i:P[r]||n.Gt(t,r,t.s[i])&&n.Ht(t,n.$o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&(P[r]=true,t.G[t.s[i]]=r,++i);}}if(i==t.w){if(n.Nt(t,-1)&&n.Mt(t,-1)&&n.Lt(t,P,-1)){for(o=false,D=0;D<t.d;D++)if(n.Ut(t,P,u,D)){o=true;break;}if(e==1&&!o||!o&&(n.Ft(t,e),e==2))return 1;}if(i==0)break;--i,t.t[i]||(P[t.G[t.s[i]]]=false);}}return t.F.a.length;},n.Pt=function(t,e){var A,P,r,i,D,u,o,Q;if(o=0,t.K){switch(e<t.f&&n.ut(t.p,e)?o=Pt(o,yr):e<t.f&&n.nt(t.p,e)&&(o=Pt(o,2)),Q=(r=t.u[e]&V0,r==0?0:r==i0?2:r==z1?3:4),Q!=0&&(o=Pt(o,8),Q>2&&(o=Pt(o,16)),Q>3&&(o=Pt(o,32))),A=t.s[e],A<0?o=Pt(o,ln):A>0&&(o=Pt(o,on)),D=t.j[e],D){case 0:break;case 1:o=Pt(o,A0);break;case 2:o=Pt(o,tn);break;case 3:o=Pt(o,ie);break;default:o=Pt(o,Qn);}switch(P=n.Oo(t,e),P){case 0:break;case 1:o=Pt(o,Cn);break;case 2:o=Pt(o,XA);break;case 3:o=Pt(o,GA);break;default:o=Pt(o,DA);}u=t.o[e],u>0&&(o=Pt(o,x0)),u>1&&(o=Pt(o,Oe));}else {switch(e<t.f&&n.ut(t.p,e)?o=Pt(o,yr):e<t.f&&n.nt(t.p,e)?o=Pt(o,2):o=Pt(o,4),(t.u[e]&q0)!=0?o=Pt(o,eA):o=Pt(o,nA),Q=(r=t.u[e]&V0,r==0?0:r==i0?2:r==z1?3:4),Q==0?o=Pt(o,112):Q==2?o=Pt(o,104):Q==3?o=Pt(o,88):o=Pt(o,56),A=t.s[e],A==0?o=Pt(o,PA):A<0?o=Pt(o,ln):o=Pt(o,on),i=t.e[e]-n.sp(t,e)+n.np(t,e),i){case 0:o=Pt(o,1792);break;case 1:o=Pt(o,1664);break;case 2:o=Pt(o,1408);break;default:o=Pt(o,896);}switch(D=t.j[e],D){case 0:o=Pt(o,3932160);break;case 1:o=Pt(o,rA);break;case 2:o=Pt(o,pn);break;case 3:o=Pt(o,Rn);break;default:o=Pt(o,Qn);}switch(P=n.Oo(t,e),P){case 0:o=Pt(o,NA);break;case 1:o=Pt(o,IA);break;case 2:o=Pt(o,xA);break;case 3:o=Pt(o,OA);break;default:o=Pt(o,DA);}switch(u=t.o[e],u){case 0:o=Pt(o,98304);break;case 1:o=Pt(o,81920);break;default:o=Pt(o,49152);}}return o;},n.Qt=function(t,e){var A;if(A=0,e<t.g&&(n.tt(t.p,e)||t.H[e]==8)||t.H[e]==8)A|=8;else switch(n.em(t,e)){case 0:A|=16;break;case 1:A|=1;break;case 2:A|=2;break;case 3:A|=4;break;case 4:A|=32;break;case 5:A|=64;}return (t.F[e]&64)!=0?A|=256:t.K||(A|=128),e<t.g&&n.ot(t.p,e)?A|=F0:t.K||(A|=_0),A;},n.Rt=function(t){var e,A,P,r,i,D,u,o,Q,L;for(o=0,P=t,i=0,u=P.length;i<u;++i)e=P[i],e==-1&&++o;if(o!=0)for(L=t,t=z(l.d7,H,6,L.length-o,15,1),Q=0,A=L,r=0,D=A.length;r<D;++r)e=A[r],e!=-1&&(t[Q++]=e);return w.qzb(t,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),t;},n.St=function(t,e){var A,P,r,i,D,u,o,Q;for(D=0,P=e,r=0,i=P.length;r<i;++r)A=P[r],A==-1&&++D;for(Q=z(l.d7,H,6,e.length-D,15,1),o=0,u=0;u<e.length;u++)e[u]!=-1&&(Q[o]=n.lw(t.g,u)<<16|n.lw(t.H,e[u]),t.i!=null&&(Q[o]|=t.i[u]<<24),++o);return w.qzb(Q,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),Q;},n.Tt=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b;for(i=false,r=0;r<2;r++)if(A=n.Yl(t.g,r,e),L=t.G[A],n.Zo(t.H,L)>2){for(h=n.Yl(t.g,1-r,e),o=St,u=0;u<n.Zo(t.g,A);u++)P=n.Yo(t.g,A,u),P!=h&&o>P&&(o=P);for(b=t.G[h],Q=St,D=0;D<n.Zo(t.H,L);D++)s=n.Yo(t.H,L,D),s!=b&&Q>s&&(Q=s);t.G[o]!=Q&&(i=!i);}return i;},n.Ut=function(t,e,A,P){var r,i,D,u,o,Q,L,s;for(u=t.e[P],o=u+1;o<t.A&&t.f[t.s[o]]==P;)++o;for(L=u;L<o;L++)A[L]=-1;for(D=u;;){if(s=t.u[D]==-1?t.H.f:n.Lo(t.H,t.G[t.u[D]]),++A[D],A[D]==s){if(A[D]=-1,D==u)break;--D,t.t[D]||(e[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);continue;}if(t.u[D]==-1)e[A[D]]||n.Gt(t,A[D],t.s[D])&&(t.G[t.s[D]]=A[D],e[A[D]]=true,++D);else {if(n.Yo(t.H,t.G[t.u[D]],A[D])>=t.H.f)continue;i=n.Yo(t.H,t.G[t.u[D]],A[D]),t.t[D]?i==t.G[t.s[D]]&&n.Ht(t,n.$o(t.H,t.G[t.u[D]],A[D]),t.v[D])&&++D:e[i]||n.Gt(t,i,t.s[D])&&n.Ht(t,n.$o(t.H,t.G[t.u[D]],A[D]),t.v[D])&&(e[i]=true,t.G[t.s[D]]=i,++D);}if(D==o){if(n.Nt(t,P)&&n.Mt(t,P)&&n.Lt(t,e,P)){for(Q=u;Q<o;Q++)t.t[Q]||(r=t.s[Q],e[t.G[r]]=false,t.G[r]=-1);return  true;}--D,t.t[D]||(e[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);}}return  false;},n.Vt=function(t){return n.Ot(t,1,t.c)>0;},n.Wt=function(t,e){var A;for(A=0;A<e.length;A++)if(e[A]==t)return  true;return  false;},n.Xt=function(t,e){var A,P,r;for(r=0,P=0;P<t.length;P++){for(A=t[P];e[r]<A;)if(++r,r==e.length)return  false;if(e[r]>A)return  false;}return  true;},n.Yt=function(t,e){var A,P,r,i,D,u,o,Q;if(D=false,n.Po(t.g,e)==0)for(i=1;i<n.Zo(t.g,e);i++)for(u=0;u<i;u++)P=n.Yo(t.g,e,i),r=n.Yo(t.g,e,u),t.G[P]>t.G[r]^P>r&&(D=!D);else for(i=0;i<n.Zo(t.g,e);i++){for(A=n.Yo(t.g,e,i),Q=0,o=z(l.d7,H,6,3,15,1),u=0;u<n.Zo(t.g,A);u++)o[Q]=n.Yo(t.g,A,u),o[Q]!=e&&++Q;Q==2&&t.G[o[0]]>t.G[o[1]]^o[0]>o[1]&&(D=!D);}return D;},n.Zt=function(t,e){var A,P,r,i;for(r=0,i=0;r<t.length&&i<e.length;){if(A=t[r],P=e[i],A==P)return  true;A<P?++r:++i;}return  false;},n.$t=function(t,e){var A,P,r,i,D;if(!e||e.q==0||!e.K){t.g=null;return;}for(t.g=e,t.r=false,n.aw(t.g,1),t.N=7,P=0;P<t.g.f;P++)kt(G(n.Nl(t.g,P),{l:Q0,m:0,h:3}),0)&&(t.N=15);for(D=0;D<t.g.g;D++)(n.gm(t.g,D)&se)!=0&&(t.N=15);for(t.L&&t.N!=7&&n.aw(t.H,t.N),t.p=0,t.q=0,t.D=z(l.zeb,ht,6,t.g.f,16,1),r=0;r<t.g.f;r++)t.D[r]=n.Im(t.g,r),t.D[r]&&++t.p;if(t.d=0,t.f=null,t.i=null,t.p!=0){for(i=0;i<t.g.g;i++)(t.D[n.Yl(t.g,0,i)]||t.D[n.Yl(t.g,1,i)])&&++t.q;for(A=0;A<t.g.q;A++)n.rn(t.g,A,t.D[A]);t.f=z(l.d7,H,6,t.g.q,15,1),t.d=n.gp(t.g,t.f,true,false);}},n._t=function(t,e){t.i=e;},n.au=function(t,e,A){n.bu(t,A),n.$t(t,e);},n.bu=function(t,e){if(!e||e.q==0){t.H=null;return;}t.H=e,t.L=false,n.aw(t.H,1);},n.cu=function(t,e){t.L||(n.fu(t,e),t.L=true),t.r||(n.du(t,e),n.Jt(t),n.It(t),t.r=true);},n.du=function(t,e){var A,P,r,i,D,u,o,Q,L,s;for(i=null,o=null,D=null,n.aw(t.g,t.N),t.o=z(l.d7,H,6,t.g.f,15,1),P=0;P<t.g.f;P++)t.o[P]=n.Zo(t.g,P);if(t.p!=0){for(Q=new n.tw(t.g.q,t.g.r),s=z(l.zeb,ht,6,t.g.q,16,1),r=0;r<t.g.q;r++)s[r]=!t.D[r];for(n.Bo(t.g,Q,s,true,null),n.aw(Q,t.N),n.eu(t,Q,e),i=t.j,o=t.n,D=t.k,L=0,A=0;A<t.g.f;A++)t.D[A]||(t.o[A]=n.Zo(Q,L++));}if(n.eu(t,t.g,e),t.p!=0){for(L=0,A=0;A<t.g.q;A++)t.D[A]||(t.j[A]=i[L],t.k[A]=D[L++]);for(L=0,u=0;u<t.g.r;u++)!t.D[n.Yl(t.g,0,u)]&&!t.D[n.Yl(t.g,1,u)]&&(t.n[u]=o[L++]);}},n.eu=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;for(g=e.f,t.j=z(l.e7,n0,6,e.f,14,1),t.k=z(l.d7,H,6,e.f,15,1),u=0;u<g;u++)t.j[u]=An(G(Pt(n.Pt(e,u),e.B[u]),Ur),W8),t.k[u]=e.C[u],(A&1)!=0&&(t.k[u]+=e.s[u]+16<<8),(A&2)!=0&&(t.k[u]+=e.A[u]<<16);for(t.B=z(l.e7,n0,6,e.f,14,1),m=(n.aw(e,7),e.p),c=0;c<m.j.a.length;c++){for(b=false,Q=U.wj(m.k,c),L=0,s=Q.length;L<s;++L)if(h=Q[L],(e.G[h]&Le)!=0){b=true;break;}if(!b)for(C=U.wj(m.k,c).length,r=U.wj(m.j,c),i=0,D=r.length;i<D;++i)P=r[i],C==3?t.B[P]=Pt(t.B[P],Se):C==4?t.B[P]=Pt(t.B[P],qe):C==5?t.B[P]=Pt(t.B[P],Ve):C==6?t.B[P]=Pt(t.B[P],Ee):C==7&&(t.B[P]=Pt(t.B[P],Ne));}for(d=e.g,t.n=z(l.d7,H,6,e.g,15,1),o=0;o<d;o++)t.n[o]=(n.Qt(e,o)|e.G[o])&6291967^6291840,(A&4)!=0?(t.n[o]&2)!=0&&(t.n[o]|=8):(A&8)!=0&&(t.n[o]&2)!=0&&o<e.g&&n.ot(e.p,o)&&(t.n[o]|=8);},n.fu=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b;for(n.aw(t.H,t.N),L=t.H.f,t.J=z(l.d7,H,6,L,15,1),t.I=z(l.e7,n0,6,L,14,1),D=0;D<L;D++)t.I[D]=An(G(Pt(n.Pt(t.H,D),n.Nl(t.H,D)),Ur),W8),t.J[D]=n.Sl(t.H,D),(e&1)!=0&&(t.J[D]+=n.Al(t.H,D)+16<<8),(e&2)!=0&&(t.J[D]+=n.Ll(t.H,D)<<16);for(t.M=z(l.e7,n0,6,L,14,1),h=n.Dp(t.H),Q=0;Q<h.j.a.length;Q++)for(b=U.wj(h.k,Q).length,P=U.wj(h.j,Q),r=0,i=P.length;r<i;++r)u=P[r],b==3?t.M[u]=Pt(t.M[u],Se):b==4?t.M[u]=Pt(t.M[u],qe):b==5?t.M[u]=Pt(t.M[u],Ve):b==6?t.M[u]=Pt(t.M[u],Ee):b==7&&(t.M[u]=Pt(t.M[u],Ne));for(A=0;A<L;A++)b=n.To(t.H,A),b==0?t.M[A]=Pt(t.M[A],Qe):b>7&&(t.M[A]=Pt(t.M[A],un));for(s=t.H.g,t.K=z(l.d7,H,6,s,15,1),o=0;o<s;o++)t.K[o]=(n.Qt(t.H,o)|n.gm(t.H,o))&6422527^6291840;},n.gu=function(t,e,A,P,r,i,D){var u,o;return u=n.Yo(t.g,t.s[e],P),u<t.D.length&&(!t.D[u]||t.f[u]==D)&&u!=t.u[e]&&(o=n.$o(t.g,t.s[e],P),!i[o]&&!n.Bm(t.g,o)&&(t.s[++A]=u,t.u[A]=t.s[e],t.v[A]=o,i[o]=true,r[u]?t.t[A]=true:r[u]=true)),A;},n.hu=function(){this.c=8,this.O=new U.Twb(new lt.V3());},n.iu=function(t){this.c=t,this.O=new U.Twb(new lt.V3());},n.ju=function(t,e){var A;return A=z(l.d7,H,6,e,15,1),T.Vjb(t,0,A,0,j.Math.min(t.length,e)),A;},n.ku=function(t,e){var A;return A=z(l.zeb,ht,6,e,16,1),T.Vjb(t,0,A,0,j.Math.min(t.length,e)),A;},x(64,1,{},n.hu,n.iu),y.c=0,y.d=0,y.p=0,y.q=0,y.r=false,y.w=0,y.A=0,y.L=false,y.N=0,y.P=false,l.U7=X(64),n.lu=function(){},x(215,1,{215:1},n.lu),y.a=0,y.b=0,y.c=0,y.d=0,l.S7=X(215),n.ou=function(){n.ou=o1,n.mu=F(k(l.Sbb,1),st,2,6,["QM@HzAmdqjF@","RF@Q``","qC`@ISTAlQE`","`J@H","QM@HzAmdqbF@","qC`@ISTAlQEhqPp@","sJP@DiZhAmQEb","RF@QPvR@","QM@HzA@","qC`@ISTAlQEhpPp@","qC`@Qz`MbHl","sJP@DiZhAmQEcFZF@","RFPDXH","qC`@IVtAlQE`","QM@HvAmdqfF@","sGP@DiVj`FsDVM@","`L@H","sJP@DizhAmQEcFBF@","sJP@DjvhAmQEb","sFp@DiTt@@AlqEcP","sGP@LdbMU@MfHlZ","QMHAIhD","QM@HzAy@","sJP@DkVhAmQEb","sNp@DiUjj@[\\QXu`","sJP@DiZhAmQEcFBF@","sGP@DjVj`FsDVM@","RFPDTH","RG@DXOH@","sGP@Divj`FsDVMcAC@","sGP@Dj}j`FsDVM@","qC`@Qz`MbHmFRF@","sNp@LdbJjj@[\\QXu`","QMHAIhGe@","QM@HzAyd`","QM`AIhD","qC`@ISTA@","sGP@DkUj`FsDVM@","qC`@IVtAlQEhqPp@","sNp@DiUjj@[\\QXuqea`@","KAx@@IRjuUPAlHPfES\\","QM`BN`P","sJP@DjZhAmQEcFJF@","Hid@@DjU^nBBH@FtaBXUMp`","sNp@Diujj@[\\QXuq`a`@","sJP@DjvhAmQEcFZF@","sJP@DjZhAmQEcFFF@","sOp@DjWkB@@FwDVM\\YhX@","sNp@Dj}Zj@[\\QXu`","sNp@DiWjj@[\\QXuq`a`@","sOp@DjWkB@@D","KAx@@ITouUPAlHPfES\\","KAx@@YIDTjjh@vDHSBin@","sNp@DkUZj@[\\QXu`","RFPDXOH@","QM`BN`^L`","qC`@ISTAy@","sGP@LdbMU@MfHl[FVF@","qCb@AIZ`H","KAx@@IRjuUPAlHPfES]FFa`@","KAx@@ITnuUPAlHPfES\\","HiD@@DiUVjj`AmHPfES\\H","sNp@DjUjj@[\\QXu`","sJP@DkVhAmQEcFJF@","sGP@DjVj`FsDVMcCC@","qC`@Qz`MbHmFBF@","sJP@DkfhAmQEb","qC`@IVtAlQEhsPp@","sGP@Djuj`FsDVM@","sGP@Dj}j`FsDVMcMC@","sJP@DiZhA@","KAx@@ISjuUPAlHPfES]F@a`@","sJP@DjZhAmQEcFRF@","KAx@@IRnuUPAlHPfES]F@a`@","HiD@@DjWvjj`AmHPfES\\H","QMHAIhGd@","sNp@DiUjj@[\\QXuq`a`@","KAx@@IVjmUPAlHPfES\\","sGP@DjVj`FsDVMcMC@","QM`AIhGe@","HiD@@LdbJRjjh@[RDIaTwB","qCp@AIZ`H","sGP@LdbMU@MfHl[FFF@","QMDARVA@","sNp@LdbJjj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqca`@","sGP@Dkej`FsDVM@","qCb@AIZ`OI@","HaD@@DjUZxHH@AlHPfES]FLa`@","sGP@DkYj`FsDVM@","qCb@AIV`H","sNp@LdbJjj@[\\QXuqea`@","sGP@DkUj`FsDVMcEC@","sFp@DiTt@@Axa@","Hmt@@DjU_ZxHHj@AmhPfES\\Lj","QM`BN`^P","qCb@AIZ`OH`","sFp@DiTt@@AxaP","sGP@Djuj`FsDVMcEC@","sGP@Djuj`FsDVMcIC@","sGP@DkUj`FsDVMcKC@","sJP@DkfhAmQEcFRF@","sGP@DjVj`FsDVMcIC@","HaD@@DjUZxHH@AlHPfES]FFa`@","qC`@IRtDVqDV@","sNp@Dj}Zj@[\\QXuqfa`@","KAx@@ITnuUPAlHPfES]FFa`@","HiD@@DkUUjj`AmHPfES\\H","sJQ@@dkU@H","qC`@Qz`H","KAx@@IUkmUPAlHPfES\\","KAx@@ITouUPAlHPfES]FJa`@","sJP@H~j@[TQX`","sGP@DjZj`FsDVM@","sJP@DkVhAmQEcFFF@","sJX@@eKU@H","sJP@DizhAy@","QMHAIhGbP","KAx@@ITouUPAlHPfES]FNa`@","HaD@@DjUZxHD@AlHPfES\\","HaD@@DjUZxHH@A@","sNp@LdbJjj@[\\QXuqaa`@","Hed@@LdbRQUUUP@vTHSBinFP","KAx@@ITouUPAlHPfES]FLa`@","sNp@DkUZj@[\\QXuqba`@","KAx@@ITjuUPAlHPfES]FNa`@","KAx@@YIDTjjh@vDHSBincGPp@","HaD@@DjYvxH`@AlHPfES]FLa`@","RF@QP`","qCb@AIj`H","sNp@DjUjj@[\\QXuqaa`@","sNp@DkVZj@[\\QXu`","KAx@@YIDUJjh@vDHSBin@","sGP@DkYj`FsDVMcIC@","sGP@DjVj`FsDVMcAC@","sGP@DiVj`D","sJP@DkVhAmQEcFZF@","sNp@LdbLjj@[\\QXu`","QM@HvAmdqbF@","HaD@@DjWjXHB@AlHPfES\\","sNp@DjwZj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqda`@","sFp@DiTt@@Axa`","HiD@@Djuujj`AmHPfES\\H","sNp@DkUZj@[\\QXuqca`@","sJP@DiZhAy@","KAx@@YIDTjjh@vDHSBincCPp@","KAx@@IWNmUPAlHPfES\\","KAx@@IVkMUPAlHPfES\\","sJQ@@dju@H","qCb@AIZ`OH@","qC`@ISTAxa@","sNp@DjyZj@[\\QXu`","Hid@@DjUfaBB`@FtaBXUMp`","HiD@@DiUVjj`AmHPfES\\LXBF@","KAx@@IUjmUPAlHPfES\\","HiD@@DjWvjj`AmHPfES\\LXjF@","sJP@DjVhAmQEb","qCb@AIV`OH`","HiD@@LdbJRjjh@[RDIaTwCFDa`@","KAx@@YIDTjjh@vDHSBinc@Pp@","sNp@DjUjj@[\\QXuqda`@","qC`@Qz`OED","sJP@DkfhAmQEcFZF@","KAx@@YIDbjjh@vDHSBincDPp@","sGP@Djyj`FsDVMcMC@","KAx@@IVrmUPAlHPfES\\","qCp@AIZ`OI@","sJX@@dkU@H","sJQ@@dkU@OH`","sNp@Di]ZjBBvxbqk@","Hkl@@DjU_Uk``bj`@[VDIaTwCJzX","sGP@DjZj`FsDVMcEC@","Hid@@DjU^nBBH@FtaBXUMpqcHX@","sNp@DkeZj@[\\QXu`","sNp@DjYjj@[\\QXuqca`@","sGQ@@djuT@`","HiD@@LdbJTjjh@[RDIaTwB","sOp@DjWkB@@Gd`","HeT@@LdbbRKBDQD@CYPaLJfxY@","qCr@XIKTA@","HiD@@DjW^jj`AmHPfES\\LXJF@","HeT@@DjU]k``b`@[JDIaTwCH","sGP@Djuj`FsDVMcCC@","`IH`B","sOp@DjWkB@@GdX","sJQ@@eKU@H","KAx@@YIDUJjh@vDHSBincBPp@","sJX@@eKU@OH@","KAx@@YIDTjjh@vDHSBincAPp@","sOq@@drm\\@@@`","KAx@@IUkMUPAlHPfES\\","qCp@AIj`H","Hed@@DjUUjjj@FraBXUMpr","sGX@@eJuT@`","sGP@DkUj`FsDVMcCC@","HiD@@Dj}Ujj`AmHPfES\\LXrF@","KAx@@ITouUPAlHPfES]FHa`@","Hed@@DjWujjj@FraBXUMpsFIa`@","sGP@DiUj``mfHlZ","sFp@DiTvjhAlqEcP","Hid@@DjU^nBBH@FtaBXUMpq`XX@","sJP@DkVdAmQEb","qCp@AIZ`OH`","QMhDRVA@","qC`@ISJAlQE`","qCp@BOTAyhl","sJX@@eOU@ODB","sFp@DiTt@@AyaB","sGP@DkUj`FsDVMcMC@","Hid@@DjYUaBH`@FtaBXUMpqcHX@","qC`@Qz`OH@","HiD@@DjUVjj`AmHPfES\\LXZF@","sJP@H~j@[TQXqda`@","sJX@@eKU@OI@","sNp@Djejj@[\\QXu`","sJQ@@dsU@H","sJQ@@dkU@OI`","KAx@@YIMDVjh@vDHSBin@","Hid@@DjU^nBBD@FtaBXUMp`","sNp@DkgZj@[\\QXuqca`@","qC`@IRtDVqDVcEC@","Hed@@LdbRQeUUP@vTHSBinFP","sNp@DiUjj@P","qC`@IRtDT","sNp@DkYZj@[\\QXuqca`@","KAx@@IUkmUPAlHPfES]FDa`@","KAx@@IVjmUPAlHPfES]FNa`@","sOx@@drm\\@@@`","KAx@@ITjuUPAlHPfES]FBa`@","QMDARVAyH","sJP`@dfvhA@","HeT@@DjU_k``b`@[JDIaTwCLXfF@","KAx@@IToUUPAlHPfES]FJa`@","sGP@DkYj`FsDVMcEC@","qCb@AIZ`ODH","`I@`B","KAx@@IUzmUPAlHPfES]FFa`@","sNp@DkfZj@[\\QXu`","KAx@@ITnuUPAlHPfES]F@a`@","HiD@@LddURjjh@[RDIaTwB","sNp@Dj~Zj@[\\QXuqfa`@","Hed@@Dj{uZjj@FraBXUMpr","KAx@@ITsUUPAlHPfES\\","Hid@@LdbRQk``b@AmHPfES\\LXrF@","sOp@DjWkB@@GdH","sJQ@@dkU@OH@","Hid@@DjU^nBBH@FtaBXUMpqahX@","sGP@DiYj``mfHlZ","KAx@@IToUUPAlHPfES]FLa`@","qCp@AJZ`ODH","Hmt@@DjU]ZxHHj@AmhPfES\\Lj","sGP@DkUjPFsDVM@","qC`@IVtA@","Hed@@LdbJReUUP@vTHSBinFP","sNp@DjuZj@[\\QXuqea`@","KAx@@IUkmUPAlHPfES]FNa`@","HiD@@DkVUjj`AmHPfES\\H","Hed@@DkUeZjj@FraBXUMpr","sNp@DkVZj@[\\QXuqea`@","sJP@DiVhHKZbKFLLL@","HiD@@Djuyjj`AmHPfES\\H","sNp@DjUjj@[\\QXuq`a`@","HeT@@DjYUXPbH`@[JDIaTwCH","HiD@@DjwUjj`AmHPfES\\LXRF@","sNq@@djmUPB","KAx@@YIEEZjh@vDHSBincCPp@","sGP@Di^V`dmfHlZ","Hid@@DjYUaBHP@FtaBXUMp`","sNp@DjYjj@[\\QXuqba`@","sGP@Dkej`FsDVMcKC@","HeT@@DjU^k``b`@[JDIaTwCH","qC`@Qv`MbHmFBF@","sGQ@@djmT@`","qCr@XIKTAyH","qC`@IVtAlQEhpPp@","Hid@@LdbbQxXF@@AmHPfES\\LXjF@","sGP@DkYj`FsDVMcCC@","KAx@@IVsMUPAlHPfES\\","qCp@AIj`ODl","HiD@@DkeUjj`AmHPfES\\H","deT@@DjU[k``b`@vTHSBinFP","sJP@DkVdAmQEcFRF@","HiD@@LdbJTjjh@[RDIaTwCFDa`@","HiD@@DkYyjj`AmHPfES\\H","sJP@DjZhAyH","KAx@@IVkMUPAlHPfES]FDa`@","sJX@@dkU@OI@","Hed@@LdbRQUUUP@vTHSBinFXpLL@","Hed@@DjuUZjj@FraBXUMpr","sGP@Djfj`FsDVMcKC@","sNp@DkVZj@[\\QXuqba`@","sNp@DjyZj@[\\QXuqfa`@","qCb@AIj`OH@","sNp@DjUZj@[\\QXu`","KAx@@IWOMUPAlHPfES\\","Hid@@DjU^nBBH@D","Hed@@DjuvZjj@FraBXUMpr","sJP@DiVhHKZbKFLtL@","dmt@@DjU_ZxHHj@C[PaLJfxYVLKC@","sNp@DjuZj@[\\QXuqca`@","sJP@DkfhAmQEcFJF@","sNp@LdbJZj@[\\QXu`","HeT@@DjU_k``b`@[JDIaTwCLXFF@","KAx@@IVlmUPAlHPfES]FNa`@","HeT@@LdbbRKBDQD@CYPaLJfxYcEPp@","Hid@@DjUZnBBH@FtaBXUMpqcHX@","qCa@CIKTA@","HiD@@Dj~]jj`AmHPfES\\LXFF@","sKP@Di\\Zj@[TQX`","sGP@Djfj`FsDVMcEC@","HiD@@DkgYjj`AmHPfES\\H","sNp@DjuZj@[\\QXuqaa`@","KAx@@YIMDVjh@vDHSBincDPp@","sJP@DjVhHKZbKFLTL@","Hid@@LdbRQk``b@AmHPfES\\LXZF@","HiD@@Dj}Ujj`AmHPfES\\LXzF@","HeT@@DjU_k``bP@[JDIaTwCH","sNp@DkUZi@[\\QXu`","HiD@@DjYfjj`AmHPfES\\H","sGP@DjZj`FsDVMcAC@","Hmt@@DjU_jxHHj@AmhPfES\\Lj","Hid@@LdbRQk``R@AmHPfES\\H","KAx@@YIDUJjh@vDHSBincDPp@","qCr@XIKTAyD","sOq@@drm\\@@@|`@","Hed@@DjW^jjj@FraBXUMpsFBa`@","HeT@@DjY]zXFB@@[JDIaTwCH","Hkl@@DjU_Vk``bj`@[VDIaTwCJzX","Hid@@DjY}nBHH@FtaBXUMpqcHX@","sGX@@eKuT@|d@","sGP@Dj^Y`FsDVM@","HcL@@DjU_ZnBBJh@FqaBXUMprn`","sJP@DkVdAmQEcFJF@","sOq@@drm\\@@@|b@","sNp@DjyZj@[\\QXuqaa`@","HaD@@DjUZxHH@AyD@","qC`@Qv`H","dmt@@DjU_ZxHHj@C[PaLJfxYVLYC@","sGP@Dkej`FsDVMcMC@","Hed@@DjUUjjj@FraBXUMpsFHa`@","HeT@@LdbbRkBDQD@CYPaLJfxY@","KAx@@IU{MUPAlHPfES]FLa`@","RG@DTH","sJY@DDeVhA@","KAx@@YIDUJjh@vDHSBinc@Pp@","sJX@@dkU@OI`","sJQ@@dju@OI`","HeT@@LdbbRKBDQD@CYPaLJfxYcFPp@","sFp@DiTvjhAlqEcXpPp@","HaD@@DjUZxHH@AyG@","sNx@@eJ}UPB","sNp@LddUjj@[\\QXuqca`@","HaDH@@RVU[j@@@D","sNp@DkgZi@[\\QXu`","sGY@LDeVj`D","sNp@LdbJfZBZvxbqk@","sJP`@dfvhAyL","sGX@AddQjhAxe`","Hmt@@DjU_ZxHHj@AmhPfES\\LkFIa`@","qCh@CIKTA@","sNp@LdbLjj@[\\QXuq`a`@","sOq@@drm\\@@@|a@","KAx@@IUzmUPAlHPfES]FJa`@","sNx@AddQUUPB","sGP@Di]jP`mfHlZ","sJP`@TeZhA@","KAx@@IRjmUPHKXPaLJfx","HeT@@LdbRTM\\DDT@CYPaLJfxY@","HaF@@@Rfu[j@@@D","Hid@@DjYUaBH`@FtaBXUMpqchX@","KAx@@IUjmTpAlHPfES\\","Hid@@DjU^nBBD@FtaBXUMpqcHX@","sGP@DiUj``mfHl[FFF@","KAx@@IUvmUPAlHPfES]FLa`@","Hed@@LdbQTUUUP@vTHSBinFXqDL@","sJP@DkVhA@","sOx@@drm\\@@@|b@","KAx@@IUkMUPAlHPfES]FDa`@","HeT@@LdbRQU\\DDT@CYPaLJfxY@","HiD@@Dj}Yjj`AmHPfES\\LXrF@","HiD@@Dj{ujj`AmHPfES\\LXFF@","KAx@@IWNmUPAlHPfES]FFa`@","KAx@@IRkMUPHKXPaLJfx","sJP@DjYdAmQEcFZF@","sJY@LDeZhAyL","HaDH@@RVU[f@@@D","sJP`@deVhAyB","HaD@@DjWjZjj`AlHPfES\\","sGP@DkYj`FsDVMcMC@","sNp@DkgZj@[\\QXuqea`@","sJQ@@dlu@H","HeT@@DjU]k``b`@[JDIaTwCLXrF@","sJX@@dkU@OH`","RFDDQFCr`","sJP@DiYXIKZbKFLLL@","KAx@@YIHjjjh@vDHSBincGPp@","Hk\\@@DjU^ukmLHH@@@AmXPfES\\Lki`","sGQ@@djmT@|b@","Hid@@DjUfaBB`@FtaBXUMpqahX@","sNx@@eRmUPB","Hmt@@LdbRVak``ah@FvaBXUMprh","qCr@XIJtA@","KAx@@IWMmUPAlHPfES]FNa`@","HeT@@DjYYZPbJ@@[JDIaTwCH","sNp@DkfZj@[\\QXuqea`@","Hid@@DjU^nBAHAEVtaBXUMp`","dmt@@DjUgZDHJJ@C[PaLJfxYT","sGP@DkejPFsDVM@","sNx@@eJmUPB","qCb@AIf`H","HcL@@DjU_VnBBJh@FqaBXUMprnqcXX@","Hid@@DjUZnBBH@FtaBXUMpqahX@","sNp@LdbQZjBBvxbqkcGC@","sOx@@drm\\@@@|c@","sJP@H~j@^R@","KAx@@YIDcFjhDElHPfES\\","Hid@@DjUZnBAH@FtaBXUMp`","sNp@LddUji@[\\QXu`","sGP@DjfjPFsDVM@","HeT@@DjYUXPbD`@[JDIaTwCH","KAx@@IUoMUPAlHPfES]FDa`@","sFp@DiTt@@AyaD","Hed@@DjuuZjj@FraBXUMpsFIa`@","HeT@@DjUghP`h`@[JDIaTwCLXfF@","sOp@DjWkjj`FwDVM\\YhX@","sGP@Djfj`FsDVMcIC@","KAx@@IRkmUPHKXPaLJfzL]C@","sNx@@djmUPB","QM`AIdD","sOp@DjWkB@@Gbe@","sNp@DjyZj@[\\QXuqca`@","QM@HuAmd`","sNp@LddUjj@[\\QXuqea`@","daD@@DjUZxHD@CXPaLJfzLYC@","qCb@AIZPH","HiD@@LdbJTjjh@[RDIaTwCF@a`@","Hmt@@DjU_ZxHHi@AmhPfES\\Lj","HaDH@@RYWih@H@D","HiD@@LdbJTjjh@[RDIaTwCFHa`@","sGX@@djuT@|a@","sNp@DkfZj@[\\QXuqaa`@","Hid@@DjU^nBBH@GdL","KAx@@IVkMUPAlHPfES]FJa`@","qCr@XIKTAy@","HmT@@Dj{uVjjh@[ZDIaTwCJqaXX@","dmt@@DjYWVDHbJ@C[PaLJfxYT","Hif@@@RUe^Fh@@@P","HaDH@@Rfu[j@@@GdH","KAx@@IVsMUPAlHPfES]FDa`@","sKP@Di\\Zj@[TQXq`a`@","sJX@@eMU@OH@","HeT@@DjU^k``b`@[JDIaTwCLXFF@","Hmt@@LdbbRJXPbHh@FvaBXUMprh","sJP@DjvhAmQEcFBF@","dmt@@LdbbRNXPbHh@MmBDpj[aeXplL@","sJP`@dfvhAyD","sGP@Di^V`dmfHl[FVF@","KAx@@IVsmUPAlHPfES]FBa`@","sOq@@drm\\@@@|PP","sJY@BDeZhA@","HeT@@LdbRbmBDED@CYPaLJfxY@","Hed@@Djy[Zjj@FraBXUMpr","HeT@@DjU]k``b`@[JDIaTwCLXFF@","Hid@@DjUfaBB`@D","qCa@CIJtA@","QMPARVA@","Hid@@DjUfaBB`@FtaBXUMpqcHX@","sJY@BDfZhA@","HeT@@DjUghP`hP@[JDIaTwCH","Hed@@Dj{uZjj@FraBXUMpsFIa`@","dmt@@DjYUvDHbJ@C[PaLJfxYVLUC@","sNp`@dfuZj@P","sJQ@@dmU@OH@","sJX@@dmU@H","HeT@@DjU]k``b`@[JDIaTwCLXZF@","HiD@@LdfbJZjh@[RDIaTwCFAa`@","sOx@@drm\\@@@|a@","deT@@DjUfhP`h`@vTHSBinFP","dmt@@DjU]ZxHHj@C[PaLJfxYVLKC@","sOp@DjWkjj`FwDVM\\XHX@","HcL@@LdbbRNSBDQEP@McBDpj[ae]cFpp@","HiD@@Dj}Yji`AmHPfES\\H","HaDH@@RYe[hB@@D","Hid@@DjU^njjj@FtaBXUMpq`XX@","deT@@LdbbRKBDQB@FraBXUMpr","QMPARZA@","sOq@@drm\\@@@|QX","HaD@@DjYvxH`@A@","HcL@@LdbbRNcBDQEP@McBDpj[ae]@","QMhDRZA@","RG@DXLHmP","QM`BN`XQYd","RG@DTLHmP","QMHAIXFEVd","QMDARVAaH","RFPDXLHmP","RF@Q`vRbdLEC@","RF@QpvR@","QO@HyjAmd`","`II@B","`II@CFspqJp","`II@CF[@hM@prB`","`H@[T[|B`XN@PdM@p|@bHrBcDk@","RG@DXMj}F@","QM`BN`[L~b@","RG@DTMj}D@","QMHAIXFt~j@","QMDARVA}L@","RFPDXMj}D@","sKP@Di\\YZ@[TQXqaa`@","eF`BHD"]);},n.pu=function(t,e){var A,P;if(!e)return null;for(P=z(l.d7,H,6,(n.mu.length+31)/32|0,15,1),e=n.uu(e),n.bu(t.g,e),A=0;A<n.mu.length;A++)n.$t(t.g,n.nu[A]),n.Ot(t.g,1,4)>0&&(P[A/32|0]|=1<<31-A%32);return P;},n.qu=function(t,e){var A,P;if(!e)return null;for(P=z(l.e7,n0,6,(n.mu.length+63)/64|0,14,1),e=n.uu(e),n.bu(t.g,e),A=0;A<n.mu.length;A++)n.$t(t.g,n.nu[A]),n.Ot(t.g,1,4)>0&&(P[A/64|0]=Pt(P[A/64|0],b1(1,63-A%64)));return P;},n.ru=function(){var t,e;if(n.nu==null)for(e=new n.$q(false),n.nu=z(l.d8,xe,25,n.mu.length,0,1),t=0;t<n.mu.length;t++)n.nu[t]=n.Oq(e,n.mu[t]),n.aw(n.nu[t],1);},n.su=function(t){var e;if(t.f!=null){for(e=0;e<t.f.length;e++)if((t.c[e]&~t.f[e])!=0)return  false;}else return  false;return n.tu(t);},n.tu=function(t){return !t.d&&(t.d=n.Qq(new n.$q(false),t.e)),!t.a&&(t.a=n.Qq(new n.$q(false),t.b)),n.bu(t.g,t.d),n.$t(t.g,t.a),n.Vt(t.g);},n.uu=function(t){var e,A;if(t.K){for(e=0;e<t.q;e++)if(kt(G(t.B[e],r0),0)){for(t=new n.uw(t),A=e;A<t.q;A++)kt(G(t.B[A],r0),0)&&(t.C[A]=-1);n.ul(t);}}return t;},n.vu=function(t,e,A){t.b=null,t.a=e,A==null?t.c=n.pu(t,e):t.c=A;},n.wu=function(t,e,A){t.e=null,t.d=e,A==null?t.f=n.pu(t,e):t.f=A;},n.xu=function(){n.ou(),this.g=new n.hu(),n.ru();},n.yu=function(t){n.ou();var e,A,P,r,i,D;if(t==null)return null;for(e=z(l._6,J,6,t.length*8,15,1),P=0;P<t.length;P++)for(D=t[P],r=7;r>=0;r--)A=D&15,A>9&&(A+=7),e[P*8+r]=48+A<<24>>24,D>>=4;return T.cjb((i=e,V1.Hmb(),i));},n.zu=function(t){n.ou();var e,A,P,r;if(w.Xzb(t).length==0||(w.Xzb(t).length&7)!=0)return null;for(P=z(l.d7,H,6,w.Xzb(t).length/8|0,15,1),A=0;A<w.Xzb(t).length;A++)r=A/8|0,e=(w.Szb(A,w.Xzb(t).length),w.Xzb(t).charCodeAt(A)-48),e>16&&(e-=7),P[r]<<=4,P[r]+=e;return P;},n.Au=function(t,e){n.ou();var A,P,r,i;for(i=0,P=0,r=0,A=0;A<t.length;A++)i+=Pn(t[A]&e[A]),P+=Pn(t[A]),r+=Pn(e[A]);return i/j.Math.sqrt(P*r);},n.Bu=function(t,e){n.ou();var A,P,r;for(r=0,A=0,P=0;P<t.length;P++)r+=Pn(t[P]&e[P]),A+=Pn(t[P]|e[P]);return r/A;},x(179,1,{},n.xu),l.T7=X(179),n.Du=function(t){t.j=1,t.k=10,t.n=400,t.f=400,t.i=true,t.d="black",t.b=new U.Lj(),t.a=new U.Lj(),t.c=new T.Pjb(),t.e=new F1.Nfb(12);},n.Eu=function(t,e){var A,P;return A=(P=F1.Lfb(t.e,e),new z0.dgb(P)).b,A;},n.Fu=function(t){return t.k;},n.Gu=function(t){var e,A,P,r,i,D;if(i='<svg id="'+n.Nu(t.g!=null?t.g:"mol"+n.Cu)+J1+'xmlns="http://www.w3.org/2000/svg" version="1.1" width="'+t.n+'px" height="'+t.f+'px" viewBox="0 0 '+t.n+" "+t.f+`">
`,D=t.i?"<style> #"+n.Nu(t.g!=null?t.g:"mol"+n.Cu)+" { pointer-events:none; } #"+n.Nu(t.g!=null?t.g:"mol"+n.Cu)+` .event  { pointer-events:all; } line { stroke-linecap:round; } polygon { stroke-linejoin:round; } </style>
`:'<g style="font-size:'+t.k+'px; fill-opacity:1; stroke-opacity:1; fill:black; stroke:black; font-weight:normal; text-rendering:optimizeLegibility; font-family:sans-serif; stroke-linejoin:round; stroke-linecap:round; stroke-dashoffset:0;">',i+="  ",i+=D,t.i){for(r=new U.wqb(t.b);r.a<r.c.a.length;)P=U.vqb(r),n.Hu(t,P);for(A=new U.wqb(t.a);A.a<A.c.a.length;)e=U.vqb(A),n.Hu(t,e);}return t.i||n.Hu(t,"</g>"),i+t.c.a+"</svg>";},n.Hu=function(t,e){T.Njb(t.c,"  "),T.Njb(t.c,e),T.Njb(t.c,p1);},n.Iu=function(t,e,A){n._c(),n.de.call(this,t,e),n.Du(this),this.g=A,++n.Cu;},n.Nu=function(t){var e,A,P,r,i;for(i=new T.Pjb(),r=w.Xzb(t).length,P=0;P<r;P++)switch(e=aD(t,P,w.Xzb(t).length),A=e&r1,e){case 38:i.a+="&amp;";break;case 60:i.a+="&lt;";break;case 62:i.a+="&gt;";break;case 34:i.a+="&quot;";break;case 39:i.a+="&apos;";break;default:i.a+=String.fromCharCode(A);}return i.a;},x(259,122,{122:1},n.Iu),y.pb=function(t){var e;e='<line x1="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.a),(p.ymb(),p.umb))))+J1+'y1="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.c),p.umb)))+J1+'x2="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.b),p.umb)))+J1+'y2="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.d),p.umb)))+J1+sr+this.d+J1+fP+this.j+DP,n.Hu(this,e);},y.qb=function(t){var e,A;e=T.Phb(p.pkb(p.lkb(new p.qkb(3*this.j),(p.ymb(),p.umb)))),A='<line stroke-dasharray="'+e+","+e+J1+'x1="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.a),p.umb)))+J1+'y1="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.c),p.umb)))+J1+'x2="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.b),p.umb)))+J1+'y2="'+T.Phb(p.pkb(p.lkb(new p.qkb(t.d),p.umb)))+J1+sr+this.d+J1+"stroke-width:"+this.j+DP,n.Hu(this,A);},y.rb=function(t){var e,A;for(A=new T.Rjb('<polygon points="'),e=0;e<t.a;e++)T.Ljb(A,w1(j.Math.round(t.b[e]))),A.a+=",",T.Ljb(A,w1(j.Math.round(t.c[e]))),A.a+=" ";T.Njb(A,'" fill="'+this.d+J1+sr+this.d+J1+fP+this.j+DP),n.Hu(this,A.a);},y.sb=function(t,e,A){var P,r,i,D;i=(P=(D=F1.Lfb(this.e,t),new z0.dgb(D)).b,P),r='<text x="'+T.Phb(p.pkb(p.lkb(new p.qkb(e-i/2),(p.ymb(),p.umb))))+J1+'y="'+T.Phb(p.pkb(p.lkb(new p.qkb(A+this.k/3),p.umb)))+J1+'stroke="none" font-size="'+this.e.b+J1+'fill="'+n.Nu(this.d)+'">'+n.Nu(t)+"</text>",n.Hu(this,r);},y.tb=function(t,e,A){var P;P='<circle cx="'+T.Phb(p.pkb(p.lkb(new p.qkb(t+A/2),(p.ymb(),p.umb))))+J1+'cy="'+T.Phb(p.pkb(p.lkb(new p.qkb(e+A/2),p.umb)))+J1+'r="'+T.Phb(p.pkb(p.lkb(new p.qkb(A/2),p.umb)))+J1+'fill="'+this.d+DP,n.Hu(this,P);},y.ub=function(t){return n.Eu(this,t);},y.vb=function(){return n.Fu(this);},y.xb=function(t,e,A,P){var r,i;r=n.Kl(this.L,t),i='<circle id="'+n.Nu(this.g!=null?this.g:"mol"+n.Cu)+":Atom:"+t+J1+$8+(r==0?"":'data-atom-map-no="'+r+J1)+'cx="'+T.Phb(p.pkb(p.lkb(new p.qkb(A),(p.ymb(),p.umb))))+J1+'cy="'+T.Phb(p.pkb(p.lkb(new p.qkb(P),p.umb)))+J1+'r="8'+J1+a8,U.rj(this.a,i);},y.yb=function(t,e,A,P,r){var i;i='<line id="'+n.Nu(this.g!=null?this.g:"mol"+n.Cu)+":Bond:"+t+J1+$8+'x1="'+T.Phb(p.pkb(p.lkb(new p.qkb(e),(p.ymb(),p.umb))))+J1+'y1="'+T.Phb(p.pkb(p.lkb(new p.qkb(A),p.umb)))+J1+'x2="'+T.Phb(p.pkb(p.lkb(new p.qkb(P),p.umb)))+J1+'y2="'+T.Phb(p.pkb(p.lkb(new p.qkb(r),p.umb)))+J1+fP+8+J1+a8,U.rj(this.b,i);},y.zb=function(t){this.j=nn(w1(j.Math.round(100*j.Math.max(t,1))))/100;},y.Ab=function(t){this.d="rgb("+((t&W0)>>16)+","+((t&J0)>>8)+","+(t&255)+")";},y.Bb=function(t){this.k!=t&&(this.k=t,this.e=new F1.Nfb(t));},y.Cb=function(t,e){return this.n=Gt(w1(j.Math.round(t.b))),this.f=Gt(w1(j.Math.round(t.a))),n.$d(this,t,e);},y.ob=function(){return n.Gu(this);},y.f=0,y.i=false,y.j=0,y.k=0,y.n=0,n.Cu=0,l.V7=X(259),n.Yu=function(t,e){var A;if(t.c!=null){for(A=0;A<t.c.length;A++)if(e==t.c[A]&&t.d[A])return  true;}return  false;},n.Zu=function(t,e,A,P,r){this.a=t,this.b=e,this.g=A,this.f=P,this.e=r;},x(184,1,{184:1},n.Zu),y.a=0,y.b=0,y.e=false,y.f=false,y.g=0,l.Z7=X(184),n.$u=function(t,e){!t.b&&(t.b=new lt.d4()),lt.Z3(t.b,T.uib(e));},n._u=function(t,e,A,P){var r,i,D,u,o,Q;if(r=n.Tk(e,t.d),n.hn(e,r,t.e),n.qn(e,r,t.o,false),n.fn(e,r,t.a),kt(t.c,0)&&(kt(G(t.c,2),0)?(t.c=G(t.c,-3),e.u[r]|=A0):e.u[r]&=aA,n.un(e,r,t.c,true)),t.b){for(Q=z(l.d7,H,6,t.b.d.a.length,15,1),o=0;o<t.b.d.a.length;o++)Q[o]=lt._3(t.b,o).a;n.on(e,r,Q),w.pzb(t.b.d.a,0);}else if(String.fromCharCode(A).toLowerCase().charCodeAt(0)==A&&T.ehb(A)){if(t.d!=5&&t.d!=6&&t.d!=7&&t.d!=8&&t.d!=15&&t.d!=16&&t.d!=33&&t.d!=34)throw Ut(new T.bC("SmilesParser: atomicNo "+t.d+" must not be aromatic. Position:"+(P-1)));e.u[r]|=A0;}else e.u[r]&=aA;if(t.f)for(u=new U.wqb(t.f);u.a<u.c.a.length;)D=U.vqb(u),t.r=true,n.Bn(D,0,0),n.Nn(e,true),n.cl(e,D,r);return t.g!=-1&&t.d!=1&&(i=z(l._6,J,6,1,15,1),i[0]=t.g<<24>>24,n.mn(e,r,i)),r;},n.av=function(t,e){var A;for(A=0;e<t.length&&(t[e]!=93||A!=0);)t[e]==91?++A:t[e]==93&&--A,++e;if(e==t.length)throw Ut(new T.bC("SmilesParser: No matching closing bracket found."));return e+1;},n.bv=function(t,e,A){var P,r;if(e<3||(P=e-1,(t[P]==43||t[P]==45)&&--P,!ee(t[P]&r1))||(--P,t[P]!=68&&t[P]!=82&&t[P]!=88&&t[P]!=122))return  false;for(A[0]=0;P>0&&tu(t[P-1]);)--P,++A[0];for(r=e+1;tu(t[P]);){if(t.length<=r||t[P]!==t[r])return  false;++P,++r;}return  true;},n.cv=function(t,e,A,P,r,i){var D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;if(e[A-1]==36){if(t.r=true,t.q=new U.Lj(),A+=n.gv(t,e,A-1,t.q)-1,e[A++]!=93){if(!i)throw Ut(new T.bC("SmilesParser: A positive recursive SMARTS followed by another one or by atom query features is not supported. Position:"+(A-1)));(t.k&128)==0&&(A=n.av(e,A));}return A;}if(e[A-1]==42)t.d=6,t.c=Pt(t.c,1);else if(e[A-1]==63)t.d=0;else if(Q=e[A-1]==33,Q&&(t.r=true,t.c=Pt(t.c,1),++A),e[A-1]==82&&r&&(ee(e[A]&r1)||t.j&&e[A]==123))t.d=6,t.c=Pt(t.c,1),--A,Q&&--A;else {if(L=new n.iv(),!n.dv(t,e,A-1,P,L))throw Ut(new T.bC("SmilesParser: Unexpected character in atom definition:'"+String.fromCharCode(e[A-1]&r1)+t9+(A-1)));if(t.d=L.a,A+=L.b-1,(t.k&3)!=2&&(t.g=9),r&&(e[A]==44||Q)){for(b=L.d,h=L.c,v=A-L.b;v<P;){if(!n.dv(t,e,v,P,L)){if(!Q)throw Ut(new T.bC("SmilesParser: Unexpected character in atom list:'"+String.fromCharCode(e[v]&r1)+e9+v));break;}if(L.a==1){if(!Q)throw Ut(new T.bC("SmilesParser: Hydrogen is not supported in positive atom lists:'"+T.bjb((U.Bqb(e,v,P),U.Gqb(e,z(l._6,J,6,P-v,15,1),v,P)))+e9+v));}else n.$u(t,L.a),b=b|L.d,h=h|L.c;if(v+=L.b,Q&&e[v]!=59&&e[v]!=38||!Q&&e[v]!=44||Q&&e[v+1]!=33)break;++v,e[v]==33&&++v;}t.b&&t.b.d.a.length>1&&(t.g=-1,h?b||(t.c=Pt(t.c,4)):t.c=Pt(t.c,2)),A=v;}}for(g=new n.mv(e),m=z(l.d7,H,6,1,15,1),C=true;C;){if(e[A]==64){++A,e[A]==64&&(t.i=true,++A),t.p=true;continue;}if(e[A]==58){for(++A;ee(e[A]&r1);)t.o=10*t.o+e[A]-48,++A;continue;}if(e[A]==91)throw Ut(new T.bC("SmilesParser: nested square brackets found. Position:"+A));if(e[A]==93){++A,C=false;continue;}if(t.e=n.fv(e,A,m),m[0]!=0){A+=m[0],t.e==0&&(t.c=Pt(t.c,PA));continue;}if(o=e[A]==33,o&&++A,e[A]==72){++A,A+=n.jv(g,A,1),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,128)),g.c<=1&&g.b>=1&&(D=Pt(D,256)),g.c<=2&&g.b>=2&&(D=Pt(D,512)),g.c<=3&&g.b>=3&&(D=Pt(D,i0)),o?(t.c=Pt(t.c,D),t.g=-1):g.b==g.c?t.g=g.c:(t.c=Pt(t.c,G(Z0,te(D))),t.g=-1);continue;}if(e[A]==68||e[A]==100){++A,A+=n.jv(g,A,1),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,A0)),g.c<=1&&g.b>=1&&(D=Pt(D,Ue)),g.c<=2&&g.b>=2&&(D=Pt(D,ye)),g.c<=3&&g.b>=3&&(D=Pt(D,se)),g.c<=4&&g.b>=4&&(D=Pt(D,F0)),et(D,0)!=0&&(o?t.c=Pt(t.c,D):kt(G(t.c,ae),0)?t.c=G(t.c,te(D)):(D=An(D,ae),t.c=Pt(t.c,D)));continue;}if(e[A]==122&&t.j){++A,A+=n.jv(g,A,4),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,Cn)),g.c<=1&&g.b>=1&&(D=Pt(D,n9)),g.c<=2&&g.b>=2&&(D=Pt(D,A9)),g.c<=3&&g.b>=3&&(D=Pt(D,P9)),g.c<=4&&g.b>=4&&(D=Pt(D,fA)),et(D,0)!=0&&(o?t.c=Pt(t.c,D):kt(G(t.c,Ln),0)?t.c=G(t.c,te(D)):(D=An(D,Ln),t.c=Pt(t.c,D)));continue;}if(e[A]==88){if(++A,A+=n.jv(g,A,1),S=(n.Qk(),n.Lk)[t.d],S==null)continue;M=S[0],s=n.fv(e,A,m),m[0]!=0&&(n.po(t.d)?M+=s:t.d==6?M-=j.Math.abs(s):M-=s),D=0,M-g.c<=0&&M-g.b>=0&&(D=Pt(D,x0)),M-g.c<=1&&M-g.b>=1&&(D=Pt(D,Oe)),M-g.c<=2&&M-g.b>=2&&(D=Pt(D,q1)),et(D,0)!=0&&(o?t.c=Pt(t.c,D):kt(G(t.c,$e),0)?t.c=G(t.c,te(D)):(D=An(D,$e),t.c=Pt(t.c,D)));continue;}if(e[A]==65||e[A]==97){t.c=Pt(t.c,o^e[A]==65?4:2),++A;continue;}if(e[A]==82){++A,A+=n.jv(g,A,3),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,8)),g.c<=1&&g.b>=1&&(D=Pt(D,16)),g.c<=2&&g.b>=2&&(D=Pt(D,32)),g.c<=3&&g.b>=3&&(D=Pt(D,64)),g.b>3&&n.Gv(t.n,(o?"!R":"R")+g.b),et(D,0)!=0&&(o?t.c=Pt(t.c,D):kt(G(t.c,120),0)?t.c=G(t.c,te(D)):(D=An(D,120),t.c=Pt(t.c,D)));continue;}if(e[A]==114){if(++A,A+=n.jv(g,A,1),g.a){o?t.c=Pt(t.c,384):t.c=Pt(t.c,8);continue;}d=g.c,g.b>g.c&&n.Gv(t.n,(o?"!r":"r")+("{"+g.c+"-"+g.b+"}")),!o&&d>=3&&d<=7?t.c=Pt(t.c,l.Oeb(d<<22)):g.b>g.c||n.Gv(t.n,(o?"!r":"r")+d);continue;}if(e[A]==118){++A,A+=n.jv(g,A,1),M=g.c,g.b>g.c&&n.Gv(t.n,(o?"!v":"v")+("{"+g.c+"-"+g.b+"}")),!o&&M<=14?t.a=M:g.b>g.c||n.Gv(t.n,(o?"!v":"v")+M);continue;}if(e[A]==94){if(++A,u=e[A++]-48,u<1||u>3)throw Ut(new T.bC("SmilesParser: Unsupported hybridization. Position:"+A));c=u==1?q1:u==2?Oe:x0,o||(c=G($e,te(c))),t.c=Pt(t.c,c);continue;}if(e[A]==36){if(!o)throw Ut(new T.bC("SmilesParser: non-negated recursive SMARTS relating to preceding atom are not supported yet. Position:"+A));!t.f&&(t.f=new U.Lj()),t.r=true,A+=n.gv(t,e,A,t.f);continue;}if(r&&(e[A]==59||e[A]==38)){t.r=true,++A;continue;}if(r&&e[A]==44&&n.bv(e,A,m)){t.r=true,A+=m[0]+1;continue;}if(r&&e[A]==44&&(t.k&128)!=0){t.r=true,A+=1;break;}throw e[A]==44?Ut(new T.bC("SmilesParser: alternative atom definitions not supported. (Tip: enumerate SMARTS): '"+String.fromCharCode(e[A]&r1)+hr+A)):Ut(new T.bC("SmilesParser: unexpected character inside brackets: '"+String.fromCharCode(e[A]&r1)+hr+A));}return A;},n.dv=function(t,e,A,P,r){var i,D;if(r.d=true,r.c=true,e[A]==35){for(++A,t.r=true,r.a=0,r.b=1;A<P&&ee(e[A]&r1);)r.a=10*r.a+e[A]-48,++r.b,++A;if(r.a==0||r.a>=(n.Qk(),n.Kk).length)throw Ut(new T.bC("SmilesParser: Atomic number out of range. position:"+(A-1)));return  true;}if(e[A]>=65&&e[A]<=90){if(r.b=e[A+1]>=97&&e[A+1]<=122?2:1,r.a=(n.Qk(),n.oo(T.Bjb(w.vzb(e,A,(D=r.b,V1.Hmb(),D))),321)),r.b==2&&r.a==0&&(r.b=1,r.a=n.oo(T.Bjb(w.vzb(e,A,(i=r.b,i))),321)),r.d=false,r.a==0)throw Ut(new T.bC("SmilesParser: Unknown atom label. position:"+(A-1)));return  true;}return e[A]==65&&e[A+1]==115||e[A]==83&&e[A+1]==101?(r.b=2,r.a=(n.Qk(),n.oo(T.Bjb(w.vzb(e,A,(i=r.b,V1.Hmb(),i))),321)),r.c=false,true):e[A]==99||e[A]==110||e[A]==111||e[A]==112||e[A]==115?(r.b=1,r.a=(n.Qk(),n.oo(T.Bjb(w.vzb(e,A,(i=r.b,V1.Hmb(),i))),321)),r.c=false,true):false;},n.ev=function(t,e,A,P,r){if(e[A-1]==42)t.d=6,t.c=Pt(t.c,1);else if(e[A-1]==63)t.d=0;else if((e[A-1]==65||e[A-1]==97)&&r)t.d=6,t.c=Pt(t.c,1),t.c=Pt(t.c,e[A-1]==65?4:2),t.r=true;else switch(String.fromCharCode(e[A-1]).toUpperCase().charCodeAt(0)){case 66:A<P&&e[A]==114?(t.d=35,++A):t.d=5;break;case 67:A<P&&e[A]==108?(t.d=17,++A):t.d=6;break;case 70:t.d=9;break;case 73:t.d=53;break;case 78:t.d=7;break;case 79:t.d=8;break;case 80:t.d=15;break;case 83:t.d=16;}return A;},n.fv=function(t,e,A){var P,r;if(A[0]=0,t[e]==43||t[e]==45){for(r=t[e],P=1,++A[0];t[e+A[0]]==r;)++P,++A[0];return P==1&&ee(t[e+1]&r1)&&(P=t[e+1]-48,++A[0]),r==43?P:-P;}return 0;},n.gv=function(t,e,A,P){var r,i,D,u,o;if(e[A+1]!=40)throw Ut(new T.bC("SmilesParser: '$' for recursive SMARTS must be followed by '('. position:"+A));for(u=1,i=A+2;i<e.length&&u>0;)e[i]==40?++u:e[i]==41&&--u,++i;if(u>0)throw Ut(new T.bC("SmilesParser: Missing closing ')' for recursive SMARTS. '('-position:"+(A+1)));if(D=new n.tw(16,16),o=new n.Iv(t.k),n.Ev(o,t.n.f),n.wv(o,D,e,A+2,i-1,true,true),n.Nn(D,true),w.nzb(P.a,D),e[A-1]==33)for(r=0;r<D.q;r++)n.un(D,r,r0,true);return i-A;},n.hv=function(t,e){this.n=t,this.k=e,this.j=(e&16)==0,this.d=-1,this.e=0,this.o=0,this.a=-1,this.g=-1,this.c=0;},x(283,1,{},n.hv),y.a=0,y.c=0,y.d=0,y.e=0,y.g=0,y.i=false,y.j=false,y.k=0,y.o=0,y.p=false,y.r=false,l.Y7=X(283),n.iv=function(){this.a=-1;},x(363,1,{},n.iv),y.a=0,y.b=0,y.c=false,y.d=false,l.W7=X(363),n.jv=function(t,e,A){var P,r,i,D,u;if(t.a=false,t.d=e,ee(t.e[e]&r1)){for(u=n.kv(t),t.c=t.b=u,P=e-1;P>1&&t2(t.e[P-1]);)--P;for(;t.e[t.d]==44;){for(D=true,i=e-P,r=0;r<i;r++)if(t.e[P+r]!==t.e[t.d+1+r]){D=false;break;}if(!D)break;t.d+=1+i,u=n.kv(t),t.c>u?t.c=u:t.b<u&&(t.b=u);}return t.d-e;}return t.e[e]==123&&ee(t.e[e+1]&r1)?(++t.d,t.c=n.kv(t),t.e[t.d++]!=45||!ee(t.e[t.d]&r1)||(t.b=n.kv(t),t.e[t.d++]!=125)?0:t.d-e):(t.c=1,t.b=A,t.a=true,0);},n.kv=function(t){var e;return e=t.e[t.d++]-48,ee(t.e[t.d]&r1)&&(e=10*e+(t.e[t.d++]-48)),e;},n.lv=function(t){return "{"+t.c+"-"+t.b+"}";},n.mv=function(t){this.e=t;},x(364,1,{},n.mv),y.ob=function(){return n.lv(this);},y.a=false,y.b=0,y.c=0,y.d=0,l.X7=X(364),n.ov=function(t,e,A,P,r,i){var D;return D=n.Vk(t.k,e,A,P),U.wnb(t.s,e+"_"+A,F(k(l.d7,1),H,6,15,[r,i])),U.wnb(t.s,A+"_"+e,F(k(l.d7,1),H,6,15,[r,i])),U.vrb(t.a,e),U.vrb(t.a,A),D;},n.pv=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c;for(Q=z(l.d7,H,6,t.k.f,15,1),u=z(l.d7,H,6,t.k.f,15,1),o=z(l.d7,H,6,t.k.f,15,1),L=z(l.d7,H,6,t.k.f,15,1),A=n.Yl(t.k,0,e),P=n.Yl(t.k,1,e),u[0]=A,u[1]=P,o[0]=-1,o[1]=e,Q[A]=1,Q[P]=2,L[A]=-1,L[P]=A,D=1,s=1;D<=s&&Q[u[D]]<15;){for(c=u[D],h=0;h<n.Zo(t.k,c);h++)if(r=n.Yo(t.k,c,h),r!=L[c]){if(i=n.$o(t.k,c,h),r==A){for(o[0]=i,b=0;b<=s;b++)t.g[o[h]]||(t.g[o[h]]=true,++t.c);return;}n.Jm(t.k,r)&&Q[r]==0&&(++s,u[s]=r,o[s]=i,Q[r]=Q[c]+1,L[r]=c);}++D;}},n.qv=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b;for(n.aw(t.k,7),s=false,h=z(l.d7,H,6,2,15,1),b=z(l.d7,H,6,2,15,1),L=z(l.d7,H,6,2,15,1),P=0;P<t.k.g;P++)if(!n.iq(t.k,P)&&n.hm(t.k,P)==2){for(D=0;D<2;D++){for(h[D]=-1,L[D]=-1,e=n.Yl(t.k,D,P),Q=0;Q<n.Zo(t.k,e);Q++)r=n.$o(t.k,e,Q),r!=P&&(h[D]==-1&&(n.hm(t.k,r)==257||n.hm(t.k,r)==129)?(h[D]=n.Yo(t.k,e,Q),b[D]=r):L[D]=n.Yo(t.k,e,Q));if(h[D]==-1)break;}if(h[0]!=-1&&h[1]!=-1){for(o=n.hm(t.k,b[0])==n.hm(t.k,b[1]),u=0;u<2;u++)h[u]==n.Yl(t.k,0,b[u])&&(o=!o);for(i=0;i<2;i++)L[i]!=-1&&L[i]<h[i]&&(o=!o);n.In(t.k,P,o?2:1,false),s=true;}}for(A=0;A<t.k.g;A++)(n.hm(t.k,A)==257||n.hm(t.k,A)==129)&&n.Ln(t.k,A,1);return s;},n.rv=function(t){return t==61?2:t==35?4:t==36?32:t==58?8:t==62?16:t==126?127:1;},n.sv=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;for(A=0;A<t.k.f;A++)if(n.Jm(t.k,A))for(o=z(l.d7,H,6,t.k.f,15,1),u=z(l.d7,H,6,t.k.f,15,1),Q=z(l.d7,H,6,t.k.f,15,1),u[0]=A,o[A]=1,Q[A]=-1,D=0,L=0;D<=L;){for(r=(o[u[D]]&1)==1?1:2,s=0;s<n.Zo(t.k,u[D]);s++)if(P=n.$o(t.k,u[D],s),n.em(t.k,P)==r&&e[P]&&(i=n.Yo(t.k,u[D],s),o[i]==0)){if(r==1&&n.Jm(t.k,i)){for(h=u[D];h!=-1;)n.Ln(t.k,n.Wo(t.k,i,h),r==1?2:1),r=3-r,i=h,h=Q[h];return n.rn(t.k,A,false),n.rn(t.k,i,false),t.b-=2,true;}u[++L]=i,Q[i]=u[D],o[i]=o[u[D]]+1;}++D;}return  false;},n.tv=function(t){var e,A,P,r;for(e=0;e<t.k.f;e++)if(n.Sl(t.k,e)==7&&n.Al(t.k,e)==0&&n.up(t.k,e)>3&&n.Po(t.k,e)>0){for(r=0;r<n.Zo(t.k,e);r++)if(A=n.Yo(t.k,e,r),P=n.$o(t.k,e,r),n.em(t.k,P)>1&&n.Gm(t.k,A)){n.hm(t.k,P)==4?n.Ln(t.k,P,2):n.Ln(t.k,P,1),n.hn(t.k,e,n.Al(t.k,e)+1),n.hn(t.k,A,n.Al(t.k,A)-1),n.fn(t.k,e,-1);break;}}},n.uv=function(t){return t.r?"Unresolved SMARTS features:"+t.r:"";},n.vv=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot;for(n.aw(t.k,1),t.g=z(l.zeb,ht,6,t.k.g,16,1),t.c=0,s=0;s<t.k.g;s++)n.hm(t.k,s)==8&&(n.Ln(t.k,s,1),t.g[s]=true,++t.c);for(v=z(l.zeb,ht,6,t.k.f,16,1),Ot=new n.Ct(t.k,3),C=z(l.zeb,ht,6,Ot.j.a.length,16,1),$t=0;$t<Ot.j.a.length;$t++){for(Ht=U.wj(Ot.j,$t),C[$t]=true,O=Ht,I=0,Dt=O.length;I<Dt;++I)if(S=O[I],!n.Jm(t.k,S)){C[$t]=false;break;}if(C[$t]){for(K=Ht,ut=0,Lt=K.length;ut<Lt;++ut)S=K[ut],v[S]=true;for(xt=U.wj(Ot.k,$t),V=xt,N=0,Qt=V.length;N<Qt;++N)S=V[N],t.g[S]||(t.g[S]=true,++t.c);}}for(h=0;h<t.k.g;h++)!t.g[h]&&(Ot.c[h]&r1)!=0&&n.Jm(t.k,n.Yl(t.k,0,h))&&n.Jm(t.k,n.Yl(t.k,1,h))&&n.pv(t,h);for(b=0;b<t.k.g;b++)t.g[b]||(i=n.Yl(t.k,0,b),D=n.Yl(t.k,1,b),!v[i]&&!v[D]&&n.Jm(t.k,i)&&n.Jm(t.k,D)&&(t.g[b]=true,++t.c));for(n.aw(t.k,7),(t.q==2||t.q==1&&t.p)&&n.Cv(t),m=z(l.zeb,ht,6,t.k.g,16,1),t.k.g>=0&&T.Vjb(t.g,0,m,0,t.k.g),vt=0;vt<Ot.j.a.length;vt++)if(C[vt]){for(Ht=U.wj(Ot.j,vt),rt=Ht,yt=0,bt=rt.length;yt<bt;++yt)if(ot=rt[yt],!n.Dv(t,ot))for(n.Jm(t.k,ot)&&(n.rn(t.k,ot,false),--t.b),S=0;S<n.Zo(t.k,ot);S++)g=n.$o(t.k,ot,S),t.g[g]&&(t.g[g]=false,--t.c);}for(n.Bv(t),wt=0;wt<Ot.j.a.length;wt++)if(C[wt]&&U.wj(Ot.k,wt).length==6){for(xt=U.wj(Ot.k,wt),M=true,o=xt,Q=0,L=o.length;Q<L;++Q)if(u=o[Q],!t.g[u]){M=false;break;}M&&(n.Av(t,xt[0]),n.Av(t,xt[2]),n.Av(t,xt[4]),n.Bv(t));}for(ct=5;ct>=4;ct--)do for(it=false,u=0;u<t.k.g;u++)if(t.g[u]){for(P=0,d=0;d<2;d++)for(c=n.Yl(t.k,d,u),S=0;S<n.Zo(t.k,c);S++)t.g[n.$o(t.k,c,S)]&&++P;if(P==ct){n.Av(t,u),n.Bv(t),it=true;break;}}while(it);for(;t.b>=2&&n.sv(t,m););if(e){if(t.b!=0)for(r=0;r<t.k.f;r++)n.Jm(t.k,r)&&(n.rn(t.k,r,false),n.un(t.k,r,2,true),--t.b,A=true);if(t.c!=0)for(u=0;u<t.k.g;u++)t.g[u]&&(t.g[u]=false,n.Ln(t.k,u,8),--t.c,A=true);}else for(r=0;r<t.k.f;r++)n.Jm(t.k,r)&&n.np(t.k,r)!=0&&(n.rn(t.k,r,false),n.vn(t.k,r,32),--t.b);if(t.q==0||t.q==1&&!A){if(t.b!=0)throw Ut(new T.bC(r9));if(t.c!=0)throw Ut(new T.bC(r9));}},n.wv=function(t,e,A,P,r,i,D){var u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt,mt,y1,u1,Mt,Zt,m1,Et,f1,x1,s1,Q1,I1,G1;for(t.k=e,n.$v(t.k),t.s=new U.utb(),t.a=new U.wrb(),t.r&&T.Agb(t.r,0),t.b=0,t.p=false,Q=t.q!=0,xt=null,c=z(l.d7,H,6,32,15,1),c[0]=-1,y1=z(l.d7,H,6,16,15,1),Zt=z(l.d7,H,6,16,15,1),Mt=z(l.d7,H,6,16,15,1),u1=z(l.d7,H,6,16,15,1),ot=0;ot<16;ot++)y1[ot]=-1;for(h=0,N=-1,m1=false,yt=false,Qt=false,v=0,m=1,d=0;A[P]<=32;)++P;for(;P<r;){if(Et=A[P++]&r1,T.ehb(Et)||Et==42||Et==63||Et==33&&Q&&m1||Et==35&&Q&&m1||Et==36&&Q&&m1){if(b=new n.hv(t,t.j|t.q),m1){if((t.j&128)!=0){if(V=new n.Kv(P-1),P=n.cv(b,A,P,r,true,true),A[P-1]!=93){for(;A[P-1]!=93;)P=n.cv(b,A,P+1,r,true,true),++V.a;t.f.add(V);}}else P=n.cv(b,A,P,r,Q,false);}else P=n.ev(b,A,P,r,Q);if(m1=false,b.q&&U.wj(b.q,0)){N=c[v],c[v]=e.q,n.Xk(e,b.q?U.wj(b.q,0):null),N!=-1&&m!=512&&(g=n.Vk(t.k,N,N,m),d!=0&&(t.p=true,n.Kn(t.k,g,d,true),n.Rk(t.k,g))),m=1,d=0;continue;}if(b.d==-1&&Et!=63)throw Ut(new T.bC("SmilesParser: unknown element label found. Position:"+(P-1)));b.r&&(t.p=true),s=n._u(b,t.k,Et,P),n.Jm(t.k,s)&&++t.b,N=c[v],N!=-1&&m!=512&&(g=n.Vk(t.k,N,s,m),d!=0&&(t.p=true,n.Kn(t.k,g,d,true),n.Rk(t.k,g))),m=1,d=0,c[v]=s,h!=0&&(n.sn(t.k,s,h),h=0),D&&(vt=xt?U.jpb(xt,T.uib(N)):null,vt&&n.Mv(vt,s,(bt=P,b.d==1&&n.Ll(t.k,s)==0,bt)),b.p&&(!xt&&(xt=new U.awb()),Dt=b.g==9?0:b.g,U.Uvb(xt,T.uib(s),new n.Sv(t,s,P-2,N,Dt,b.i))));continue;}if(Et==46){c[v]=-1,m=512;continue;}if(Et==45||Et==61||Et==35||Et==36||Et==58||Et==47||Et==92||Et==60||Et==126||Et==33||Et==64){if(m1)throw Ut(new T.bC("SmilesParser: unexpected bond symbol inside square brackets: '"+String.fromCharCode(Et)+hr+(P-1)));for(O=0;Et==45||Et==61||Et==35||Et==36||Et==58||Et==47||Et==92||Et==60||Et==126||Et==33||Et==64;){if(Et==33){if(Et=A[P++]&r1,Et==64)d|=128;else if(Et==45&&A[P]==62||Et==60&&A[P]==45)O|=16,++P;else if(Et==45)O|=1;else if(Et==61)O|=2;else if(Et==35)O|=4;else if(Et==36)O|=32;else if(Et==58)O|=8;else throw Ut(new T.bC("SmilesParser: bond symbol '"+String.fromCharCode(Et)+"' not allowed after '!'. Position:"+(P-1)));}else if(Et==64?d|=256:Et==61?m=2:Et==35?m=4:Et==36?m=32:Et==58?m=8:Et==126?d|=15:Et==47?D&&(m=257):Et==92?D&&(m=129):(Et==45&&A[P]==62||Et==60&&A[P]==45)&&(m=16,++P),A[P]==44)for(d|=(m==16?62:Et)==61?2:(m==16?62:Et)==35?4:(m==16?62:Et)==36?32:(m==16?62:Et)==58?8:(m==16?62:Et)==62?16:(m==16?62:Et)==126?127:1;A[P]==44;)A[P+1]==60&&A[P+2]==45||A[P+1]==45&&A[P+2]==62?(d|=16,P+=3):(d|=n.rv(A[P+1]&r1),P+=2);if(A[P]==59){++P,Et=A[P++]&r1;continue;}O!=0&&(d|=127&~O);break;}continue;}if(Et<=32){P=r;continue;}if(ee(Et)){if(ct=Et-48,m1){for(;P<r&&ee(A[P]&r1);)ct=10*ct+A[P]-48,++P;h=ct;}else {if(C=yt?P-3:P-2,ut=A[C]==45||A[C]==47||A[C]==92||A[C]==61||A[C]==35||A[C]==36||A[C]==58||A[C]==62||A[C]==126,yt&&P<r&&ee(A[P]&r1)&&(ct=10*ct+A[P]-48,yt=false,++P),ct>=y1.length){if(ct>=100)throw Ut(new T.bC("SmilesParser: ringClosureAtom number out of range: "+ct));for(wt=y1.length,it=y1.length;it<=ct;)it=j.Math.min(100,it+16);for(y1=(w.Hzb(it),U.Gqb(y1,z(l.d7,H,6,it,15,1),0,it)),Zt=(w.Hzb(it),U.Gqb(Zt,z(l.d7,H,6,it,15,1),0,it)),Mt=(w.Hzb(it),U.Gqb(Mt,z(l.d7,H,6,it,15,1),0,it)),u1=(w.Hzb(it),U.Gqb(u1,z(l.d7,H,6,it,15,1),0,it)),Lt=wt;Lt<it;Lt++)y1[Lt]=-1;}if(y1[ct]==-1)y1[ct]=c[v],Zt[ct]=P-1,Mt[ct]=ut?m:-1,u1[ct]=ut?d:0;else {if(y1[ct]==c[v])throw Ut(new T.bC("SmilesParser: ring closure to same atom"));D&&xt&&(vt=U.jpb(xt,T.uib(y1[ct])),vt&&n.Mv(vt,c[v],Zt[ct]),vt=U.jpb(xt,T.uib(c[v])),vt&&n.Mv(vt,y1[ct],P-1)),Mt[ct]!=-1?m=Mt[ct]:m==257?m=129:m==129&&(m=257),u=y1[ct],o=c[v],g=n.ov(t,u,o,m,Zt[ct],P-1),u1[ct]!=0&&(d=u1[ct]),d!=0&&(t.p=true,n.Kn(t.k,g,u1[ct],true),n.Rk(t.k,g)),y1[ct]=-1;}m=1,d=0;}continue;}if(Et==43)throw Ut(new T.bC("SmilesParser: '+' found outside brackets. Position:"+(P-1)));if(Et==40){if(c[v]==-1){Qt=true;continue;}++v,c.length==v&&(c=U.Eqb(c,c.length+32)),c[v]=c[v-1];continue;}if(Et==41){if(v==0){if(!Qt)throw Ut(new T.bC("SmilesParser: Closing ')' without opening counterpart. Position:"+(P-1)));c[0]=-1,Qt=false;continue;}--v;continue;}if(Et==91){m1=true;continue;}if(Et==93)throw Ut(new T.bC("SmilesParser: closing bracket at unexpected position:"+(P-1)));if(Et==37){yt=true;continue;}throw Ut(new T.bC("SmilesParser: unexpected character outside brackets: '"+String.fromCharCode(Et)+t9+(P-1)));}if(m!=1)throw Ut(new T.bC("SmilesParser: dangling open bond"));for(Nt=y1,pt=0,mt=Nt.length;pt<mt;++pt)if(Ot=Nt[pt],Ot!=-1)throw Ut(new T.bC("SmilesParser: dangling ring closure."));for(I=n.kp(t.k),n.Pn(t.k,true),n.aw(t.k,1),L=0;L<t.k.q;L++)if(n.Dl(t.k,L)!=null){if(K=n.El(t.k,L)[0],t.p||t.q==2){if(K!=9)if(t.i)for(Lt=0;Lt<K;Lt++)n.Vk(t.k,L,n.Tk(t.k,1),1);else K==0&&n.un(t.k,L,1792,true),K==1&&n.un(t.k,L,1664,true),K==2&&n.un(t.k,L,1408,true),K==3&&n.un(t.k,L,896,true);}else {if(K==9&&(K=0),!n.Km(t.k,L)&&(!n.Jm(t.k,L)||n.Sl(t.k,L)==6&&n.Al(t.k,L)==0)){for(G1=n.lo(n.Sl(t.k,L)),M=false,f1=n.up(t.k,L),f1-=n.lm(t.k,L,f1),f1+=K,n.Jm(t.k,L)&&++f1,s1=G1,Q1=0,I1=s1.length;Q1<I1;++Q1)if(x1=s1[Q1],f1<=x1){M=true,x1==f1+2?n.vn(t.k,L,48):x1==f1+1?n.vn(t.k,L,32):(x1!=f1||x1!=G1[0])&&n.fn(t.k,L,f1);break;}M||n.fn(t.k,L,f1);}if(t.i||!n.zq(t.k,L))for(Lt=0;Lt<K;Lt++)n.Vk(t.k,L,n.Tk(t.k,1),1);}}else !t.i&&(t.p||t.q==2)&&(K=n.bp(t.k,L),K>=1&&n.un(t.k,L,128,true),K>=2&&n.un(t.k,L,256,true),K>=3&&n.un(t.k,L,512,true),K>=4&&n.un(t.k,L,i0,true));if(!t.i&&(t.p||t.q==2)&&n.pq(t.k,false),n.aw(t.k,1),n.tv(t),n.vv(t,Q,t.p),t.k.t=null,n.Pn(t.k,false),D&&(n.qv(t),xt)){for(Ht=($t=new U.Job(xt).a.ad().Sb(),new U.Qob($t));Ht.a.cd();)vt=(S=Ht.a.dd(),S.gd()),n.tn(t.k,I[vt.a],n.Nv(vt,I),false);n.uq(t.k,0);}n.uq(t.k,0),i&&(rt=new Z.BA(t.d),kt(t.n,0)&&Z.xA(rt,t.n),Z.iA(rt,t.k),D&&n.nw(t.k)),(t.p||t.q==2)&&(n.Nn(t.k,true),n.Bq(t.k),n.Cq(t.k));},n.xv=function(t,e,A,P,r){n.wv(t,e,A,0,A.length,P,r);},n.yv=function(t,e){var A;return e==null?null:n.zv(t,w.xzb((A=e,V1.Hmb(),A)));},n.zv=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;for(D=lt.C3(e,0);D>0&&e[D-1]==45;)D=lt.C3(e,D+1);for(u=D==-1?-1:lt.C3(e,D+1);u>0&&e[u-1]==45;)u=lt.C3(e,u+1);if(u==-1)throw Ut(new T.bC("Missing one or both separators ('>')."));if(lt.C3(e,u+1)!=-1)throw Ut(new T.bC("Found more than 2 separators ('>')."));for(h=new tt.yH(),s=0,i=0,A=-1;i<e.length;){for(;i<e.length&&e[i]==46;)++i;if(e[i]==40){if(A!=-1)throw Ut(new T.bC("Second open group bracket found before closing first one."));for(++i,o=0,r=i;r<e.length;r++)if(e[r]==40)++o;else if(e[r]==41&&o--==0){A=r;break;}}for(P=i;P<e.length&&e[P]!=62&&!(e[P]==46&&(t.o&&A==-1||A==P-1||P+1==e.length||e[P+1]==46));)++P;if(L=P,A==P-1&&(--L,A=-1),i!=L){if(Q=new n.sw(),n.wv(t,Q,e,i,L,true,true),t.q==1&&t.p)return n.zv(new n.Iv(t.j|2),e);s==0?(U.rj(h.g,Q),h.d=-1):s==1?U.rj(h.a,Q):(U.rj(h.f,Q),h.d=-1);}for(i=P;i<e.length&&e[i]==62;)++i,++s;}return h;},n.Av=function(t,e){var A,P,r,i;for(n.hm(t.k,e)==1&&n.Ln(t.k,e,2),r=0;r<2;r++)for(A=n.Yl(t.k,r,e),n.Jm(t.k,A)&&(n.rn(t.k,A,false),--t.b),i=0;i<n.Zo(t.k,A);i++)P=n.$o(t.k,A,i),t.g[P]&&(t.g[P]=false,--t.c);},n.Bv=function(t){var e,A,P,r,i,D,u;do for(u=false,A=0;A<t.k.g;A++)if(t.g[A]){for(i=false,r=0;r<2;r++){for(e=false,P=n.Yl(t.k,r,A),D=0;D<n.Zo(t.k,P);D++)if(A!=n.$o(t.k,P,D)&&t.g[n.$o(t.k,P,D)]){e=true;break;}if(!e){i=true;break;}}i&&(u=true,n.Av(t,A));}while(u);},n.Cv=function(t){var e,A,P,r,i,D;for(e=0;e<t.k.g;e++)if(t.g[e]){for(i=true,r=0;r<2&&i;r++)for(A=n.Yl(t.k,r,e),D=0;D<n.Zo(t.k,A)&&i;D++)e!=n.$o(t.k,A,D)&&t.g[n.$o(t.k,A,D)]&&(i=false);if(i)for(n.Ln(t.k,e,8),--t.c,P=0;P<2;P++)A=n.Yl(t.k,P,e),n.Jm(t.k,A)&&(n.rn(t.k,A,false),--t.b);}},n.Dv=function(t,e){var A,P;return !n.Et(n.Sl(t.k,e))||n.Sl(t.k,e)==6&&(!n.Jm(t.k,e)||n.Al(t.k,e)>0)||(A=n.Dl(t.k,e)==null||n.El(t.k,e)[0]==9?0:n.El(t.k,e)[0],P=n.ip(t.k,e)-A,P<1)?false:n.Sl(t.k,e)==16||n.Sl(t.k,e)==34||n.Sl(t.k,e)==52?n.Zo(t.k,e)==2&&n.Al(t.k,e)<=0?false:P!=2:true;},n.Ev=function(t,e){t.f=e;},n.Fv=function(t,e){t.n=e;},n.Gv=function(t,e){t.e&&(!t.r&&(t.r=new T.Pjb()),T.Njb(t.r," "),T.Njb(t.r,e));},n.Hv=function(){n.Iv.call(this,0);},n.Iv=function(t){this.j=t&-4,this.q=t&3,this.o=(t&32)!=0,this.e=(t&64)!=0,this.i=(t&8)!=0,this.d=2,(t&4)!=0&&(this.d|=1),this.i&&(this.d&=-3);},x(113,1,{},n.Hv,n.Iv),y.b=0,y.c=0,y.d=0,y.e=false,y.i=false,y.j=0,y.n=0,y.o=false,y.p=false,y.q=0,l.b8=X(113),n.Jv=function(t,e){return r8(e.b,t.b);},n.Kv=function(t){this.b=t,this.a=1;},x(216,1,{216:1,32:1},n.Kv),y.Rb=function(t){return n.Jv(this,t);},y.a=0,y.b=0,l.$7=X(216),n.Mv=function(t,e,A){if(!t.c){if(t.e.a.length==4){t.c=true;return;}U.rj(t.e,new n.Tv(t,e,A));}},n.Nv=function(t,e){var A,P,r;if(t.c)return 3;for(r=new U.wqb(t.e);r.a<r.c.a.length;)P=U.vqb(r),P.a!=yA&&P.a!=St&&(P.a=e[P.a]);switch(A=false,t.e.a.length){case 2:if(U.wj(t.e,0).a>=yA||U.wj(t.e,1).a>=yA)return 3;A=n.Qv(t,e);break;case 3:U.rj(t.e,new n.Tv(t,St,t.b));case 4:A=n.Rv(t);break;default:return 3;}return t.d^A?1:2;},n.Ov=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt;if(v=t.f.k,r=U.wj(t.e,A),Dt=r.a,s=e[t.a],d=n.np(v,Dt)>0&&v.j[Dt]==v.e[Dt],g=U.srb(t.f.a,t.f.t[Dt]),!d&&!g)return D=v.i[Dt][2],v.C[D]==1&&(D=t.f.t[D],(A==0?D>Dt:t.f.t[v.i[Dt][1]]>D)&&(P=!P)),P;if(d&&(i=v.i[Dt][A],D=t.f.t[i],D>Dt&&(P=!P,v.C[i]==1&&(P=!P))),g){for(M=v.e[Dt],b=z(l.d7,H,6,M-1,15,1),c=z(l.d7,H,6,M-1,15,1),S=0,m=0,O=0;m<M;m++)D=v.i[Dt][m],D!=s&&(b[O]=D,Q=n.Wo(v,Dt,D),I=U.tnb(t.f.s,t.f.t[Dt]+"_"+t.f.t[D]),c[O++]=I==null?-1:Q,I!=null&&++S);if(d)b[0]<Dt&&(P=!P);else switch(S){case 1:if(n.np(v,Dt)>0)break;C=c[0]>=0,h=b[C?0:1],V=b[C?1:0],V>Dt?h>Dt&&v.C[V]!=1&&(P=!P):h<Dt&&(P=!P);break;case 2:u=n.Pv(t,c[0],Dt),o=n.Pv(t,c[1],Dt),ut=U.tnb(t.f.s,Dt+"_"+t.f.t[u]),Qt=U.tnb(t.f.s,Dt+"_"+t.f.t[o]),L=t.b,A==0?(K=ut[1]<L?ut[1]:ut[0],N=Qt[1]<L?Qt[1]:Qt[0]):(K=ut[0]<L?ut[1]:ut[0],N=Qt[0]<L?Qt[1]:Qt[0]),K<N!=u<o&&(P=!P);}}return P;},n.Pv=function(t,e,A){return t.f.k.D[0][e]==A?t.f.k.D[1][e]:t.f.k.D[0][e];},n.Qv=function(t,e){var A,P;if(P=false,t.f.t==null)for(t.f.t=z(l.d7,H,6,e.length,15,1),A=t.f.t.length;--A>=0;)t.f.t[e[A]]=A;return P=n.Ov(t,e,0,P),P=n.Ov(t,e,1,P),P;},n.Rv=function(t){var e,A,P;for(A=false,e=1;e<t.e.a.length;e++)for(P=0;P<e;P++)U.wj(t.e,P).a>U.wj(t.e,e).a&&(A=!A),U.wj(t.e,P).b>U.wj(t.e,e).b&&(A=!A);return A;},n.Sv=function(t,e,A,P,r,i){this.f=t,r!=0&&r!=1?this.c=true:(this.a=e,this.b=A,this.d=i,this.e=new U.Lj(),P!=-1&&n.Mv(this,P,A-1),P!=-1&&r==1&&n.Mv(this,yA,A+1));},x(217,1,{217:1},n.Sv),y.a=0,y.b=0,y.c=false,y.d=false,l.a8=X(217),n.Tv=function(t,e,A){this.c=t,this.a=e,this.b=A;},x(181,1,{181:1},n.Tv),y.ob=function(){return "["+(this.a==yA?"h":this.a==St?"lp":n.Hl(this.c.f.k,this.a))+this.b+"]";},y.a=0,y.b=0,l._7=X(181),n.Vv=function(t){t.c=new U.Lj();},n.Wv=function(t,e){var A,P,r,i;if(i=t.c.a.length,i==0)return U.qj(t.c,0,e),0;for(r=1;2*r<=i;)r<<=1;for(P=r,--r;P!=0;){if(P>>=1,r>=i){r-=P;continue;}if(A=T.$ib(e,U.wj(t.c,r)),A==0)return  -1;if(P==0)break;A<0?r-=P:r+=P;}return r<i&&T.$ib(e,U.wj(t.c,r))>0&&++r,U.qj(t.c,r,e),r;},n.Xv=function(t,e){var A,P,r,i;if(i=t.c.a.length,i==0)return  -1;for(r=1;2*r<=i;)r<<=1;for(P=r,--r;P!=0;){if(P>>=1,r>=i){r-=P;continue;}if(A=T.$ib(e,U.wj(t.c,r)),A==0)return r;if(P==0)break;A<0?r-=P:r+=P;}return  -1;},n.Yv=function(){n.Vv(this);},x(102,1,i9,n.Yv),y.ac=function(t){return n.Xv(this,t);},l.c8=X(102),n.$v=function(t){n.il(t),t.d=null;},n._v=function(t,e){n.ol(t,e),t.d&&(e.T=0);},n.aw=function(t,e){var A,P,r,i;if(n.Do(t,e),(e&~t.T)!=0){for((t.T&15)!=0&&t.q>1&&n.di(t.J[0],t.J[1])&&Z.iA(new Z.BA(0),t),t.c&&(e|=128),A=0;A<t.q;A++)t.u[A]&=-67223560;for(P=0;P<t.g;P++)t.F[P]&=-64;r=0,i=0,(e&32)!=0?(r=32,i=1):(e&64)!=0&&(r=64,i=3),(e&128)!=0&&(r|=128,i|=32),t.d=new n.Ng(t,i),n.Ig(t.d),n.Jg(t.d),n.Hg(t.d),n.rw(t)&&(t.d=new n.Ng(t,i)),t.T|=24|r;}},n.bw=function(t,e){return n.vg(t.d,e);},n.cw=function(t,e){return n.kg(t.d,e);},n.dw=function(t){var e;switch(n.aw(t,31),e=t.I&r1,t.I&fe){case q1:return null;case A0:return e==1?"meso":""+e+" meso diastereomers";case 0:return "unknown chirality";case JA:return "both enantiomers";case Ue:return "this enantiomer";case 327680:return "this or other enantiomer";case tn:return "two epimers";default:return e==1?"one stereo isomer":""+e+" stereo isomers";}},n.ew=function(t){var e;return e=new n.tw(t.q,t.r),n.nl(t,e),e;},n.fw=function(t){var e,A,P,r,i,D,u;for(n.aw(t,15),D=z(l.d7,H,6,3,15,1),A=0;A<t.f;A++)(t.u[A]&q0)!=0&&(u=(t.u[A]&U1)>>18,u!=0&&(D[u]=j.Math.max(D[u],(t.u[A]&U1)>>18!=1&&(t.u[A]&U1)>>18!=2?-1:(t.u[A]&k0)>>20)));for(r=0;r<t.g;r++)((t.F[r]&3)==1||(t.F[r]&3)==2)&&t.H[r]==1&&(u=(t.F[r]&768)>>8,u!=0&&(D[u]=j.Math.max(D[u],(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&S0)>>10)));for(i=z(l.d7,S1,7,3,0,2),i[1]=z(l.d7,H,6,1+D[1],15,1),i[2]=z(l.d7,H,6,1+D[2],15,1),e=0;e<t.f;e++)(t.u[e]&q0)!=0&&(u=(t.u[e]&U1)>>18,u!=0&&++i[u][(t.u[e]&U1)>>18!=1&&(t.u[e]&U1)>>18!=2?-1:(t.u[e]&k0)>>20]);for(P=0;P<t.g;P++)((t.F[P]&3)==1||(t.F[P]&3)==2)&&t.H[P]==1&&(u=(t.F[P]&768)>>8,u!=0&&++i[u][(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&S0)>>10]);return i;},n.gw=function(t){var e,A;return e=z(l.d7,H,6,t.q,15,1),A=n.gp(t,e,false,false),n.hw(t,e,A);},n.hw=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;for(d=z(l.d8,xe,25,A,0,1),D=z(l.d7,H,6,A,15,1),Q=z(l.d7,H,6,A,15,1),i=z(l.d7,H,6,t.q,15,1),r=0;r<t.q;r++)e[r]!=-1&&(i[r]=D[e[r]]++);for(o=0;o<t.r;o++)c=e[t.D[0][o]],g=e[t.D[1][o]],c==g&&c!=-1&&++Q[c];for(m=0;m<A;m++)d[m]=new n.tw(D[m],Q[m]),n._v(t,d[m]);for(P=0;P<t.q;P++)e[P]!=-1&&n.kl(t,d[e[P]],P,0,0);for(u=0;u<t.r;u++)c=e[t.D[0][u]],g=e[t.D[1][u]],c==g&&c!=-1&&n.ml(t,d[c],u,0,0,i,false);for(s=d,h=0,b=s.length;h<b;++h)L=s[h],n.bn(L,1),n.bn(L,2);return d;},n.iw=function(t){return n.aw(t,15),!t.d&&(t.f<2||!n.di(t.J[0],t.J[1]))&&(t.d=new n.Mg(t)),t.d?n.qg(t.d):null;},n.jw=function(t){return n.aw(t,15),!t.d&&(t.f<2||!n.di(t.J[0],t.J[1]))&&(t.d=new n.Mg(t)),t.d?n.lg(t.d):null;},n.kw=function(t){var e,A;for(n.aw(t,31),A=0,e=0;e<t.f;e++)(t.u[e]&3)!=0&&(t.u[e]&4)==0&&++A;return A;},n.lw=function(t,e){return n.ug(t.d,e);},n.mw=function(t,e){t.c=e,t.T&=-144;},n.nw=function(t){n.aw(t,31),t.d&&n.Kg(t.d);},n.ow=function(t){var e,A;for(n.aw(t,15),t.L=false,e=0;e<t.q;e++)t.u[e]&=eP,(t.u[e]&3)!=0?t.u[e]|=lA:t.u[e]&=tP;for(A=0;A<t.r;A++)(t.F[A]&3)!=0&&n.em(t,A)==2?t.H[A]=386:t.H[A]&=-385;t.T&=-249;},n.pw=function(t,e,A){var P,r,i,D,u,o,Q,L,s;if(s=t.u[e]&3,s==1||s==2){if(u=false,t.o[e]==2&&t.j[e]==2&&t.n[e][0]==2&&t.n[e][1]==2&&t.C[e]<=7)for(D=0;D<t.j[e];D++){for(P=t.i[e][D],L=0,Q=z(l.d7,H,6,3,15,1),o=0;o<t.j[P];o++)Q[L]=t.i[P][o],Q[L]!=e&&++L;L==2&&Q[0]<Q[1]^A[Q[0]]<A[Q[1]]&&(u=!u);}else for(D=1;D<t.j[e];D++)for(o=0;o<D;o++)r=t.i[e][D],i=t.i[e][o],r<i^A[r]<A[i]&&(u=!u);u&&(s=s==1?2:1);}return s;},n.qw=function(t){var e,A,P,r,i,D;for(n.Aq(t),n.aw(t,31),P=0;P<t.f;P++){if(((t.u[P]&U1)>>18==1||(t.u[P]&U1)>>18==2)&&((t.u[P]&q0)==0||(t.u[P]&3)==3))throw Ut(new T.bC(Tr));if((t.u[P]&q1)!=0)throw Ut(new T.bC(br));if(((t.u[P]&3)==1||(t.u[P]&3)==2)&&t.o[P]==0){for(e=z(l.b7,jt,6,t.j[P],15,1),i=0;i<t.j[P];i++)e[i]=n.Xl(t,P,t.i[P][i]);for(r=1;r<t.j[P];r++)if(!n.Rm(t,t.k[P][r],P)){for(D=0;D<r;D++)if(!n.Rm(t,t.k[P][D],P)&&(A=j.Math.abs(n.no(e[r],e[D])),A<.08||A>G8))throw Ut(new T.bC(cr));}}}},n.rw=function(t){var e,A,P,r,i,D,u;for(u=false,A=0;A<t.f;A++)((t.u[A]&q0)==0||(t.u[A]&3)==3)&&(t.u[A]&=eP);for(i=0;i<t.g;i++)(n.em(t,i)!=1||(t.F[i]&3)==0||(t.F[i]&3)==3)&&(t.F[i]&=-32513);if(t.L){if((t.I&fe)!=A0){for(D=z(l.zeb,ht,6,t.f,16,1),P=0;P<t.f;P++)(t.u[P]&q0)!=0&&(t.u[P]&3)!=3&&(t.u[P]&U1)>>18==1&&(D[P]=true);for(r=0;r<t.f;r++)(t.u[r]&q0)!=0&&(t.u[r]&3)!=3&&(n.nn(t,r,1,0),u=true);for(e=0;e<t.f;e++)D[e]&&(n.tn(t,e,1,false),n.nn(t,e,1,-1),u=true);}t.L=false;}return n.bn(t,1),n.bn(t,2),u;},n.sw=function(){n.Qk(),n.Eq.call(this);},n.tw=function(t,e){n.Qk(),n.Fq.call(this,t,e);},n.uw=function(t){n.Qk(),n.Gq.call(this,t);},x(25,114,f9,n.sw,n.tw,n.uw),y.Xb=function(t){n._v(this,t);},y.Zb=function(t){n.aw(this,t);},y.c=false,l.d8=X(25),n.yw=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;for(u=new U.Lj(),n.aw(e,1),Q=z(l.zeb,ht,6,e.f,16,1),A=0;A<e.f;A++)if(t.f[A]&&(n.Sl(t.i,A)==7||n.Sl(t.i,A)==8||n.Sl(t.i,A)==16||n.Sl(t.i,A)==34||n.Sl(t.i,A)==52)){for(Q[A]=true,o=0;o<e.j[A];o++)if(P=e.i[A][o],i=e.k[A][o],s=e.n[A][o],e.o[P]!=0&&e.o[A]<s){for(Q[P]=true,U.rj(u,T.uib(i)),L=0;L<e.j[P];L++)r=e.i[P][L],Q[r]||(Q[r]=true,D=e.k[P][L],h=e.n[P][L],e.o[P]+2==s+h&&(U.rj(u,T.uib(D)),s>=h&&(e.o[r]<h?n.Hw(e,r)&&n.Bw(t,e,r,true,false,Q,u):n.Bw(t,e,r,true,true,Q,u)),h>=s&&n.Hw(e,A)&&n.Bw(t,e,r,false,false,Q,u),t.f[r]&&(n.Sl(t.i,r)==5||n.Sl(t.i,r)==6||n.Sl(t.i,r)==7||n.Sl(t.i,r)==8||n.Sl(t.i,r)==16||n.Sl(t.i,r)==34||n.Sl(t.i,r)==52)&&e.o[r]<h&&(s<=2&&h>=2&&n.Hw(e,A)&&n.zw(t,e,i,D),s>=2&&h<=2&&n.Hw(e,r)&&n.zw(t,e,D,i)),U.Dj(u,u.a.length-1)),Q[r]=false);U.Dj(u,u.a.length-1),Q[P]=false;}Q[A]=false;}},n.zw=function(t,e,A,P){var r;r=new n.Nw(t,e),n.Mw(r,A,n.em(e,A)==1?2:3),n.Mw(r,P,n.em(e,P)==2?1:2),t.g[A]=true,t.g[P]=true,U.Nwb(t.e,r)&&U.Ppb(t.d,r);},n.Aw=function(t,e){U.Nwb(t.e,e)&&U.Ppb(t.d,e);},n.Bw=function(t,e,A,P,r,i,D){var u,o,Q,L,s,h,b,c,g,d,m,C;for(b=0;b<e.j[A];b++)if(u=e.i[A][b],!i[u]&&(L=e.k[A][b],m=n.em(e,L),P&&m>=2||!P&&m<=2)){for(i[u]=true,D.add(T.uib(L)),c=0;c<e.j[u];c++)if(o=e.i[u][c],!i[o]&&(s=e.k[u][c],C=n.em(e,s),e.o[u]+2==m+C&&(P&&C<=2||!P&&C>=2))){if(i[o]=true,D.add(T.uib(s)),t.f[o]&&(n.Sl(t.i,o)==5||n.Sl(t.i,o)==6||n.Sl(t.i,o)==7||n.Sl(t.i,o)==8||n.Sl(t.i,o)==16||n.Sl(t.i,o)==34||n.Sl(t.i,o)==52)&&(!P||n.Hw(e,o))){for(h=new n.Nw(t,e),g=0;g<D.a.length;g++)Q=(w.Kzb(g,D.a.length),D.a[g]).a,d=g<2?P^(g&1)==0:r^(g&1)==0,d?n.Mw(h,Q,n.em(e,Q)==1?2:3):n.Mw(h,Q,n.em(e,Q)==2?1:2),t.g[Q]=true;U.Nwb(t.e,h)&&U.Ppb(t.d,h);}else n.Bw(t,e,o,P,r,i,D);U.Dj(D,D.a.length-1),i[o]=false;}U.Dj(D,D.a.length-1),i[u]=false;}return  false;},n.Cw=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(t.b=z(l.d7,H,6,t.i.f,15,1),u=z(l.d7,H,6,t.i.f,15,1),P=z(l.zeb,ht,6,t.i.g,16,1),s=0,A=0;A<t.i.g;A++)if(!P[A]&&t.g[A])for(++s,t.b[n.Yl(t.i,0,A)]=s,t.b[n.Yl(t.i,1,A)]=s,P[A]=true,Q=0;Q<2;Q++)for(e=n.Yl(t.i,Q,A),t.b[e]=s,D=0,o=0,u[0]=e;D<=o;){for(L=0;L<n.Zo(t.i,u[D]);L++)i=n.$o(t.i,u[D],L),!P[i]&&t.g[i]&&(P[i]=true,r=n.Yo(t.i,u[D],L),t.b[r]==0&&(t.b[r]=s,u[++o]=r));++D;}t.j=s;},n.Dw=function(t){var e,A,P;for(t.n=z(l.d7,H,6,t.j,15,1),t.k=z(l.d7,H,6,t.j,15,1),t.o=z(l.d7,H,6,t.j,15,1),e=0;e<t.i.f;e++)t.b[e]!=0&&(P=t.b[e]-1,t.a!=null&&(t.k[P]+=t.a[e]),t.c!=null&&(t.o[P]+=t.c[e]));for(A=0;A<t.i.g;A++)t.g[A]&&n.em(t.i,A)==2&&(t.n[t.b[n.Yl(t.i,0,A)]-1]+=2);},n.Ew=function(t){var e,A,P,r;for(P=0;P<t.i.r;P++)for(r=0;r<2;r++)e=n.Yl(t.i,r,P),A=n.Yl(t.i,1-r,P),n.Sl(t.i,e)==1&&n.Ll(t.i,e)>1&&n.Sl(t.i,A)>1&&t.b[A]!=0&&(n.Ll(t.i,e)==2?(t.a==null&&(t.a=z(l.d7,H,6,t.i.q,15,1)),++t.a[A]):(t.c==null&&(t.c=z(l.d7,H,6,t.i.q,15,1)),++t.c[A]),n.Tm(t.i,e));(t.a!=null||t.c!=null)&&n.ul(t.i);},n.Fw=function(t){var e;for(t.e=new U.Swb(),t.d=new U._pb(),n.Aw(t,new n.Nw(t,t.i)),e=n.ew(t.i);!U.Tpb(t.d);)if(n.Lw(U.Vpb(t.d),e),n.yw(t,e),t.e.a.c>=n.xw){T.Ujb(),""+n.qg(new n.Mg(t.i));break;}},n.Gw=function(t){var e,A,P,r,i,D,u,o,Q,L;if(t.e.a.c==1)return t.i;for(Q=n.ew(t.i),n.Nn(Q,true),n.aw(Q,7),r=0;r<Q.g;r++)t.g[r]&&(Q.H[r]=1,Q.T=0,n.Kn(Q,r,3,true));for(A=0;A<Q.f;A++)t.b[A]!=0&&n.sp(t.i,A)<4&&(n.Ao(Q,A),Q.u[A]&=tP,Q.T&=7,n.nn(Q,A,0,-1));for(u=z(l.d7,H,6,t.j,15,1),o=z(l.d7,H,6,t.j,15,1),P=new n.Mg(Q).f,e=0;e<Q.f;e++)t.b[e]!=0&&(L=t.b[e]-1,o[L]<P[e]&&(o[L]=P[e],u[L]=e));for(i=0;i<t.j;i++)D=""+t.n[i]+"|"+t.k[i]+"|"+t.o[i],n.ln(Q,u[i],D);return Q;},n.Hw=function(t,e){return t.e[e]-n.sp(t,e)+n.np(t,e)<=0?false:n.Gm(t,e)?true:t.o[e]==0;},n.Iw=function(t){var e,A,P,r,i,D,u,o;n.aw(t.i,1),u=t.i.f;do--u;while(u>=0&&n.Sl(t.i,u)==1);for(e=0;e<u;e++)if(n.Sl(t.i,e)==1){n.Yn(t.i,e,u);do--u;while(n.Sl(t.i,u)==1);}if(u!=t.i.f-1){for(D=z(l.zeb,ht,6,t.i.g,16,1),i=0;i<t.i.g;i++)A=n.Yl(t.i,0,i),P=n.Yl(t.i,1,i),(n.Sl(t.i,A)==1||n.Sl(t.i,P)==1)&&(D[i]=true);o=t.i.g;do--o;while(o>=0&&D[o]);for(r=0;r<o;r++)if(D[r]){n.Zn(t.i,r,o),D[r]=false;do--o;while(D[o]);}}},n.Jw=function(t){var e,A;for(this.i=n.ew(t),n.Iw(this),n.aw(this.i,7),this.g=z(l.zeb,ht,6,this.i.g,16,1),this.f=z(l.zeb,ht,6,this.i.f,16,1),e=0;e<this.i.f;e++)A=n.lo(n.Sl(this.i,e))[0],this.f[e]=n.sp(this.i,e)<A;n.Fw(this),n.Cw(this),n.Ew(this),n.Dw(this);},x(241,1,{},n.Jw),y.j=0,n.xw=uP,l.f8=X(241),n.Kw=function(t,e){return lt.U3(t.a,e.a);},n.Lw=function(t,e){var A,P;for(P=0;P<t.b.i.g;P++)t.b.g[P]&&(A=3&t.a[P>>4]>>2*(P&15),n.Ln(e,P,A==1?1:A==2?t.b.g[P]&&!n.iq(t.b.i,P)?386:2:A==3?4:16));},n.Mw=function(t,e,A){var P,r;P=e>>4,r=2*(e&15),t.a[P]&=~(3<<r),t.a[P]|=A<<r;},n.Nw=function(t,e){var A;for(this.b=t,this.a=z(l.d7,H,6,(t.i.g+15)/16|0,15,1),A=0;A<t.i.g;A++)this.a[A>>4]|=j.Math.min(3,n.em(e,A))<<2*(A&15);},x(165,1,{165:1,32:1},n.Nw),y.Rb=function(t){return n.Kw(this,t);},l.e8=X(165),n.Pw=function(t,e,A){var P,r,i,D,u;for(r=0,D=new U.wqb(A);D.a<D.c.a.length;)i=U.vqb(D),w.Xzb(i).length!=0&&(u=R.N2(e,i).b,r<u&&(r=u));return P=t.c*UA*(A.a.length-1)+t.c,new Tt.FN(t.i[0].a,t.i[0].b-t.c/2,r,P);},n.Qw=function(t,e,A){return !!t.b&&Tt.AN(t.b,e,A);},n.Rw=function(t){var e,A,P,r,i;if(A=T.ljb(t,Z1(38)),A==-1)return t;for(r=0,i=new T.Gjb();A!=-1;)T.Fjb(i,(w.Rzb(r,A,w.Xzb(t).length),w.Xzb(t).substr(r,A-r))),w.Szb(A+1,w.Xzb(t).length),w.Xzb(t).charCodeAt(A+1)==38?(i.a+="&",r=A+2):(e=(w.Szb(A+1,w.Xzb(t).length),w.Xzb(t).charCodeAt(A+1)),e=e-(e<65?48:e<97?65:97),P=(w.Szb(A+2,w.Xzb(t).length),w.Xzb(t).charCodeAt(A+2)),P=P-(P<65?48:P<97?65:97),T.Djb(i,16*e+P&r1),r=A+3),A=T.mjb(t,Z1(38),r);return T.Fjb(i,(w.Szb(r,w.Xzb(t).length+1),w.Xzb(t).substr(r))),i.a;},n.Sw=function(t){var e,A;for(A=new T.Gjb(),e=0;e<w.Xzb(t).length;e++)switch(w.Szb(e,w.Xzb(t).length),w.Xzb(t).charCodeAt(e)){case 38:A.a+="&&";break;case 9:A.a+="&09";break;case 10:A.a+="&0A";break;case 32:A.a+="&20";break;default:T.Djb(A,(w.Szb(e,w.Xzb(t).length),w.Xzb(t).charCodeAt(e)));}return A.a;},n.Tw=function(t,e){var A;return A=n.Uw(t),n.Pw(t,e,A);},n.Uw=function(t){var e,A,P;if(A=new U.Lj(),e=T.ljb(t.e,Z1(10)),e==-1)U.rj(A,t.e);else {for(P=0;e!=-1;)U.rj(A,T.vjb(t.e,P,e)),P=e+1,e=T.kjb(t.e,10,P);U.rj(A,T.ujb(t.e,P));}return A;},n.Vw=function(t,e,A){t.i[0].a=e,t.i[0].b=A;},n.Ww=function(t,e){try{t.c=I0(e);}catch(A){if(A=H1(A),!Jt(A,40))throw Ut(A);}},n.Xw=function(t,e){try{t.d=T.Qhb(e,P1,St);}catch(A){if(A=H1(A),!Jt(A,40))throw Ut(A);}},n.Yw=function(t,e,A,P){t.e=e,t.c=A,t.d=P;},n.Zw=function(t,e){try{t.i[0].a=I0(e);}catch(A){if(A=H1(A),!Jt(A,40))throw Ut(A);}},n.$w=function(t,e){try{t.i[0].b=I0(e);}catch(A){if(A=H1(A),!Jt(A,40))throw Ut(A);}},n._w=function(){n.bx.call(this,new Tt.qN());},n.ax=function(t){var e,A,P,r;for(n._w.call(this),e=0;e!=-1&&(A=w.Xzb(t).indexOf('="',e),A!=-1);)P=(w.Rzb(e+1,A,w.Xzb(t).length),w.Xzb(t).substr(e+1,A-(e+1))),e=w.Xzb(t).indexOf('"',A+2),r=e==-1?(w.Szb(A+1,w.Xzb(t).length+1),w.Xzb(t).substr(A+1)):(w.Rzb(A+1,e,w.Xzb(t).length),w.Xzb(t).substr(A+1,e-(A+1))),T.ejb(P,"text")?this.e=n.Rw(r):T.ejb(P,"x")?n.Zw(this,r):T.ejb(P,"y")?n.$w(this,r):T.ejb(P,"size")?n.Ww(this,r):T.ejb(P,"style")&&n.Xw(this,r);},n.bx=function(t){this.e="",this.c=9,this.d=0,this.i=z(l._9,C1,55,1,0,1),this.i[0]=t;},x(94,193,{193:1,94:1},n._w,n.ax),y.Db=function(t,e){return this.a=!!this.b&&Tt.AN(this.b,t,e),this.a;},y.Eb=function(){this.a=false;},y.Fb=function(){var t;return t=new n._w(),n.Yw(t,this.e,this.c,this.d),n.Vw(t,this.i[0].a,this.i[0].b),t.g=this.g,t;},y.Gb=function(t,e){return n.Qw(this,t,e);},y.Hb=function(t,e){var A,P,r;for(P=e?e.c*this.c:this.c,R.T2(t,Ft(j.Math.round(P)),(this.d&1)!=0,(this.d&2)!=0),R.V2(t,this.g?fe:R.S2(t)?-1:O0),r=n.Uw(this),this.b=n.Pw(this,t,r),e&&n.wi(e,this.b),A=0;A<r.a.length;A++)R.I2(t,this.b.c,this.b.d+1+P*5/6+P*UA*A,(w.Kzb(A,r.a.length),r.a[A]));},y.Ib=function(t){return n.Tw(this,t);},y.Jb=function(){var t;return t=new T.Pjb(),T.Njb(t,' text="'+n.Sw(this.e)+'"'),T.Njb(t,' x="'+this.i[0].a+'"'),T.Njb(t,' y="'+this.i[0].b+'"'),this.c!=9&&T.Njb(t,' size="'+p.pkb(new p.skb(this.c,new p.Zlb()))+'"'),this.d!=0&&T.Njb(t,' style="'+this.d+'"'),t.a;},y.Kb=function(){return "text";},y.Lb=function(t){var e,A;e=(A=n.Uw(this),n.Pw(this,t,A)),R.V2(t,R.R2(t)),R.L2(t,e.c,e.d,e.b,e.a);},y.Mb=function(t){n.me(this,t),this.c*=t;},y.Ob=function(t,e){n.pe(this,t,e),this.c=this.f*t;},y.Pb=function(t,e){n.qe(this,t,e),this.f=this.c;},y.a=false,y.c=0,y.d=0,y.f=0,l.g8=X(94),n.ox=function(t){t.b=new U.Lj(),t.a=new U.Lj();},n.px=function(t,e){var A,P;return A=n.Wv(t,e),A==-1?-1:(P=t.b.a.length,U.rj(t.b,e),U.qj(t.a,A,T.uib(P)),P);},n.qx=function(){n.Yv.call(this),n.ox(this);},x(282,102,i9,n.qx),y.ac=function(t){var e;return e=n.Xv(this,t),e==-1?-1:U.wj(this.a,e).a;},l.h8=X(282),E.sx=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it;for(rt=n.Dp(t.e),yt=U.wj(rt.k,e).length,Lt=U.wj(rt.j,e),ot=U.wj(rt.k,e),N=true,m=1;m<yt;m++)if(E.Hx(t.b,ot[m])!=E.Hx(t.b,ot[0])){N=false;break;}if(N){for(i=(yt-2)*Vn/yt,C=0;C<yt;C++)E.xx(t,Lt[C],ot[C],ot[C==0?yt-1:C-1],i);return;}for(Qt=z(l.c7,w0,6,yt,15,1),r=0,v=0;v<yt;v++)Qt[v]=n.Po(t.e,Lt[v])==0?1.9106119386631804:n.Po(t.e,Lt[v])==1?2.094499897810209:Vn,r+=Qt[v];for(P=((yt-2)*Vn-r)/yt,M=0;M<yt;M++)Qt[M]+=P;for(L=z(l.c7,w0,6,yt,15,1),S=1;S<yt;S++)L[S]=L[S-1]+Vn-Qt[S];for(u=z(l.c7,w0,6,yt,15,1),D=0;D<100;D++){for(bt=0,it=0,C=0;C<yt;C++)bt+=E.Hx(t.b,ot[C])*j.Math.sin(L[C]),it+=E.Hx(t.b,ot[C])*j.Math.cos(L[C]);for(b=n.mo(0,0,bt,it),g=j.Math.sqrt(bt*bt+it*it),ut=-1,I=0,V=0;V<yt;V++)O=V==0?yt-1:V-1,K=V+1==yt?0:V+1,o=n.no(L[V],L[O]),Q=n.no(L[K],L[V]),Dt=n.no(Qt[V],Qt[K]),u[V]=2*o-2*Q+2*Dt,c=j.Math.cos(L[V]-wr-b),h=g*c-D9*u[V],j.Math.abs(h)>j.Math.abs(I)&&(I=h,ut=V);s=j.Math.exp(-5*D/100),L[ut]+=s*I;}for(d=0;d<yt;d++)O=d==0?yt-1:d-1,A=L[O]+Vn-L[d],A>X1&&(A-=X1),E.xx(t,Lt[d],ot[O],ot[d],A);},E.tx=function(t,e){var A,P,r,i,D,u,o,Q,L;for(o=n.Dp(t.e),Q=U.wj(o.k,e).length,u=U.wj(o.k,e),L=false,r=0;r<Q;r++)if(t.f[u[r]]==0){L=true;break;}if(L){for(A=(180*Q-360)/Q*.01745329238474369,D=U.wj(o.j,e),i=0;i<Q;i++)E.xx(t,D[i],u[i==0?Q-1:i-1],u[i],A);if(Q<=4)for(P=0;P<Q;P++)t.f[u[P]]+=Q;}},E.ux=function(t,e,A,P){var r,i,D,u;for(i=-1,D=-1,u=0;u<n.Ko(t.e,e);u++)if(r=n.Yo(t.e,e,u),r==A||r==P)if(D==-1)D=u;else {i=u;break;}return t.a[e][i][D];},E.vx=function(t,e,A,P){return A<P?t.a[e][P][A]:t.a[e][A][P];},E.wx=function(t,e){var A,P,r,i,D,u,o;for(P=z(l.zeb,ht,6,n.Zo(t.e,e),16,1),o=0,r=0;r<n.Zo(t.e,e);r++){for(u=0,D=-1,i=0;i<n.Zo(t.e,e);i++)P[i]||(A=n.$o(t.e,e,i),u<t.f[A]&&(u=t.f[A],D=i));if(u==0)return o;o<<=8,o+=u,P[D]=true;}return o;},E.xx=function(t,e,A,P,r){var i,D,u,o;for(i=-1,D=-1,o=0;o<n.Ko(t.e,e);o++)if(u=n.$o(t.e,e,o),u==A||u==P)if(i==-1)i=o;else {D=o;break;}t.a[e][D][i]==0&&(t.a[e][D][i]=r,t.d[e]+=r,++t.c[e]);},E.yx=function(t,e,A,P,r){var i,D,u,o,Q,L;for(i=n.Ko(t.e,e),o=n.Zo(t.e,e),Q=1;Q<i;Q++)for(D=Q<o?t.f[n.$o(t.e,e,Q)]:0,L=0;L<Q;L++)t.a[e][Q][L]==0&&(u=L<o?t.f[n.$o(t.e,e,L)]:0,(D==A&&u==P||D==P&&u==A)&&(t.a[e][Q][L]=r));},E.zx=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O;for(this.b=e,this.e=t,this.a=z(l.c7,c0,456,this.e.f,0,3),r=0;r<this.e.f;r++)for(this.a[r]=z(l.c7,{456:1,4:1,9:1,5:1,8:1},46,n.Ko(this.e,r),0,2),o=1;o<n.Ko(this.e,r);o++)this.a[r][o]=z(l.c7,w0,6,o,15,1);for(this.c=z(l.d7,H,6,this.e.f,15,1),this.d=z(l.c7,w0,6,this.e.f,15,1),this.f=z(l.d7,H,6,this.e.g,15,1),V=n.Dp(this.e),h=z(l.zeb,ht,6,V.j.a.length,16,1),n._s(V,h,z(l.zeb,ht,6,V.j.a.length,16,1),z(l.d7,H,6,V.j.a.length,15,1),true),u=3;u<=7;u++)for(m=0;m<V.j.a.length;m++)O=U.wj(V.k,m).length,O==u&&(h[m]?E.sx(this,m):O<=4&&E.tx(this,m));for(s=z(l.zeb,ht,6,this.e.f,16,1),d=0;d<V.j.a.length;d++)if(h[d])for(v=U.wj(V.j,d),M=0,S=v.length;M<S;++M)C=v[M],s[C]=true;for(i=F(k(l.d7,1),H,6,15,[0,0,1,3,6,10,15,21]),P=0;P<this.e.f;P++){if(D=n.Ko(this.e,P),D>4){for(o=1;o<D;o++)for(c=0;c<o;c++)this.a[P][o][c]=wr;this.c[P]=i[D];continue;}if(this.c[P]!=i[D])if(n.hq(this.e,P)&&(s[P]||n.To(this.e,P)<=4&&n.Po(this.e,P)>0)){if(D>2){if(this.c[P]==1)n.Sl(this.e,P)<=14?A=(u9-this.d[P])/2:A=o9+(o9-this.d[P])*.18000000715255737;else if(A=u9-this.d[P],D>3){if(this.c[P]==2){for(b=z(l.zeb,ht,6,n.Ko(this.e,P),16,1),Q=1;Q<D;Q++)for(g=0;g<Q;g++)this.a[P][Q][g]!=0&&(b[Q]=!b[Q],b[g]=!b[g]);for(L=0;L<D;L++)if(b[L]){for(g=L+1;g<D;g++)if(b[g]){this.a[P][g][L]=A;break;}break;}}A=wr;}for(o=1;o<D;o++)for(c=0;c<o;c++)this.a[P][o][c]==0&&(this.a[P][o][c]=A);}}else if(n.hq(this.e,P)&&n.To(this.e,P)<=4)switch(E.wx(this,P)){case 771:E.yx(this,P,0,3,2.0653998851776123),E.yx(this,P,0,0,1.9814722631346626);break;case 1028:E.yx(this,P,0,4,1.9797999858856201),E.yx(this,P,0,0,1.94691481878138);break;case 393987:E.yx(this,P,0,3,l9),E.yx(this,P,0,6,l9),E.yx(this,P,3,3,1.7229016938441077);break;case 459779:E.yx(this,P,0,3,gr),E.yx(this,P,0,4,gr),E.yx(this,P,0,7,gr),E.yx(this,P,3,4,1.9322539839360076);break;case 525316:E.yx(this,P,0,4,Q9),E.yx(this,P,0,8,Q9),E.yx(this,P,4,4,1.99944913298566);case 394758:E.yx(this,P,0,6,2.526099920272827);break;case 460550:E.yx(this,P,0,6,L9),E.yx(this,P,0,7,L9);break;case 526087:E.yx(this,P,0,7,y9),E.yx(this,P,0,8,y9);break;case 526344:E.yx(this,P,0,8,2.186300039291382);break;case 50529027:E.yx(this,P,3,3,2.4189000129699707);break;case 67371779:E.yx(this,P,3,4,2.2298998832702637);break;case 67372036:E.yx(this,P,4,4,2.094399929046631);break;case 101057283:E.yx(this,P,3,6,1.839926051241747),E.yx(this,P,3,3,2.9061476191098734);break;case 117834755:E.yx(this,P,3,4,2.812249087174905),E.yx(this,P,3,7,1.7910569124592968),E.yx(this,P,4,6,2.1224948975613245);break;case 134677507:E.yx(this,P,3,4,2.642428360523752),E.yx(this,P,3,8,2.027723514585844),E.yx(this,P,4,7,2.251474717631936);break;case 117900035:E.yx(this,P,3,7,2.109753935530918),E.yx(this,P,3,3,3.1052897491356646);break;case 117900292:E.yx(this,P,4,7,2.090729910747413),E.yx(this,P,4,4,2.551671293386306);break;case 134743044:E.yx(this,P,4,8,2.139250042271712),E.yx(this,P,4,4,2.3520055858942612);}else for(A=n.Sl(this.e,P)>10?U9:n.Po(this.e,P)==2?Vn:n.Zp(this.e,P,true)?s9:n.Po(this.e,P)==0?U9:s9,o=1;o<D;o++)for(c=0;c<o;c++)this.a[P][o][c]=A;}},x(374,1,{},E.zx),l.i8=X(374),E.Gx=function(){E.Gx=o1,E.Ex=F(k(l.zeb,1),ht,6,16,[false,false,false,false,false,true,true,true,true,false,false,false,false,false,false,true,true]);},E.Hx=function(t,e){return t.a[e];},E.Ix=function(t){E.Gx();var e,A;for(n.aw(t,7),this.a=z(l.c7,w0,6,t.r,15,1),this.b=z(l.c7,w0,6,t.r,15,1),e=0;e<t.r;e++)A=E.Mx(t,e),A==-1?(this.a[e]=E.Nx(t,e),this.b[e]=E.Ox(t,e)):(this.a[e]=A==-1?2.000499963760376:E.Cx[A],this.b[e]=A==-1?1:E.Dx[A]);},E.Jx=function(t,e){return e>=t.f?0:e<t.f&&n.nt(t.p,e)&&t.C[e]==6&&t.s[e]!=0?1:t.o[e];},E.Kx=function(t,e,A,P,r,i,D,u,o){var Q,L,s,h,b;return h=P<E.Ex.length&&E.Ex[P]?j.Math.min(3,i):0,b=r<E.Ex.length&&E.Ex[r]?j.Math.min(3,D):0,u=j.Math.min(7,u),o=j.Math.min(7,o),Q=u+(h<<3)+(j.Math.min(127,P)<<5),L=o+(b<<3)+(j.Math.min(127,r)<<5),s=A?8:e?8+t:t,(s<<24)+(Q<L?(Q<<12)+L:(L<<12)+Q);},E.Lx=function(t,e,A,P,r,i,D,u,o,Q){var L;return L=E.Qx(E.Kx(t,e,A,P,r,i,D,u,o)),!Q&&L==-1&&(L=E.Qx(E.Kx(t,e,A,P,r,i,D,7,7)),L==-1&&(L=E.Qx(E.Kx(t,e,A,P,r,3,3,7,7)))),L;},E.Mx=function(t,e){var A,P,r,i,D,u;return A=t.D[0][e],P=t.D[1][e],r=t.C[A],i=t.C[P],D=t.j[A],u=t.j[P],E.Lx(n.em(t,e),e<t.g&&n.ot(t.p,e),e<t.g&&(n.tt(t.p,e)||t.H[e]==8),r,i,E.Jx(t,A),E.Jx(t,P),D,u,false);},E.Nx=function(t,e){var A,P;return A=n.Sl(t,t.D[0][e]),P=n.Sl(t,t.D[1][e]),(A<(E.pz(),E.nz).length?E.nz[A]:A<E.oz.length?sA*E.oz[A]:oP)+(P<E.nz.length?E.nz[P]:P<E.oz.length?sA*E.oz[P]:oP);},E.Ox=function(t,e){var A,P;return A=n.Sl(t,t.D[0][e]),P=n.Sl(t,t.D[1][e]),(A<(E.pz(),E.nz).length?zr:.125)+(P<E.nz.length?zr:.125);},E.Px=function(){var t,e,A,P,r,i,D;if(!E.Fx)try{for(t=E.Dy("bondLengthData.txt"),A=Rt.hgb(t),e=A==null?0:T.Qhb(A,P1,St),E.Bx=z(l.d7,H,6,e,15,1),E.Cx=z(l.c7,w0,6,e,15,1),E.Dx=z(l.c7,w0,6,e,15,1),E.Ax=z(l.d7,H,6,e,15,1),r=0;r<e;r++)if(D=Rt.hgb(t),D!=null&&(i=T.sjb(D,"\\t"),i.length==4))try{E.Bx[r]=T.Qhb(i[0],P1,St),E.Cx[r]=I0(i[1]),E.Dx[r]=I0(i[2]),E.Ax[r]=T.Qhb(i[3],P1,St);}catch(u){if(u=H1(u),Jt(u,40))break;throw Ut(u);}E.Fx=!0;}catch(u){if(u=H1(u),Jt(u,76))P=u,T.LB(P,(T.Ujb(),T.Tjb),"");else throw Ut(u);}},E.Qx=function(t){var e,A,P,r;for(E.Fx||E.Px(),r=Y1,P=z1,A=0;A<13;A++){if(e=r>=E.Bx.length||t<E.Bx[r]?-1:t==E.Bx[r]?0:1,e==0)return r;r=e<0?r-P:r+P,P=P/2|0;}return  -1;},x(373,1,{},E.Ix),E.Fx=false,l.j8=X(373),E.Rx=function(t,e){t.q=E.Qy(new E.Ty(t.o,e),t);},E.Sx=function(t,e){var A,P,r,i,D,u,o,Q,L;return A=E.Yx(t,e[0]),P=E.Yx(t,e[1]),r=E.Yx(t,e[2]),i=E.Yx(t,e[3]),o=new n.ni(P.a-A.a,P.b-A.b,P.c-A.c),Q=new n.ni(r.a-P.a,r.b-P.b,r.c-P.c),L=new n.ni(i.a-r.a,i.b-r.b,i.c-r.c),D=new n.ni(o.b*Q.c-o.c*Q.b,-(o.a*Q.c-o.c*Q.a),o.a*Q.b-o.b*Q.a),u=new n.ni(Q.b*L.c-Q.c*L.b,-(Q.a*L.c-Q.c*L.a),Q.a*L.b-Q.b*L.a),-j.Math.atan2(j.Math.sqrt(Q.a*Q.a+Q.b*Q.b+Q.c*Q.c)*(o.a*u.a+o.b*u.b+o.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},E.Tx=function(t,e){return E.Wx(t,e),E.Ey(t.q,e.q);},E.Ux=function(t,e){var A;for(A=0;A<e.q;A++)n.ii(t.k[A],e.J[A]);},E.Vx=function(t,e){var A;for(A=0;A<e.k.length;A++)n.ii(t.k[A],e.k[A]);e.j==null?t.j=null:t.j=U.Fqb(e.j,e.j.length);},E.Wx=function(t,e){var A;(!t.q||!e.q)&&(A=new E.Sy(t.o),!t.q&&(t.q=E.Qy(A,t)),!e.q&&(e.q=E.Qy(A,e)));},E.Xx=function(t,e){return t.j==null?-1:t.j[e];},E.Yx=function(t,e){return t.k[e];},E.Zx=function(t){return t.p==null?t.o.P:t.p;},E.$x=function(t,e){return t.k[e].a;},E._x=function(t,e){return t.k[e].b;},E.ay=function(t,e){return t.k[e].c;},E.by=function(t,e,A){for(t.j==null&&(t.j=z(l.yeb,Kt,6,t.o.r,15,1),U.Lqb(t.j));A<0;)A=A+360<<16>>16;for(;A>=360;)A=A-360<<16>>16;t.j[e]=A;},E.cy=function(t,e,A){n.ii(t.k[e],A);},E.dy=function(t,e,A){t.k[e]=A;},E.ey=function(t,e){t.p=e;},E.fy=function(t,e,A){t.k[e].a=A;},E.gy=function(t,e,A){t.k[e].b=A;},E.hy=function(t,e,A){t.k[e].c=A;},E.iy=function(t,e){var A;for(!e&&(e=n.ew(t.o)),A=0;A<e.q;A++)n.ii(e.J[A],t.k[A]);return t.p!=null&&n.Tn(e,t.p),e;},E.jy=function(t){var e;for(this.o=t,this.k=z(l.w7,Sn,29,t.q,0,1),e=0;e<t.q;e++)this.k[e]=new n.oi(t.J[e]);this.n=NaN;},E.ky=function(t){E.ly.call(this,t,t.o);},E.ly=function(t,e){var A;for(this.o=e,this.k=z(l.w7,Sn,29,t.k.length,0,1),A=0;A<this.k.length;A++)this.k[A]=new n.oi(t.k[A]);t.j!=null&&(this.j=U.Fqb(t.j,t.j.length)),this.p=t.p==null||T.djb(t.p)?t.p:T.ajb(t.p,vn),this.n=NaN;},x(63,1,{63:1,32:1},E.ky),y.Rb=function(t){return E.Tx(this,t);},y.n=0,l.k8=X(63),E.py=function(t,e){var A,P,r,i,D;return e==null?null:(D=U.jpb(t.c,e),D||(P=w.Xzb("<").length,(T.ejb(w.Xzb(e).substr(w.Xzb(e).length-P,P),"<")||(r=w.Xzb("-").length,T.ejb(w.Xzb(e).substr(w.Xzb(e).length-r,r),"-")))&&(D=U.jpb(t.c,(i=w.Xzb("<").length,T.ejb(w.Xzb(e).substr(w.Xzb(e).length-i,i),"<")?T.vjb(e,0,w.Xzb(e).length-1)+">":(A=w.Xzb("-").length,T.ejb(w.Xzb(e).substr(w.Xzb(e).length-A,A),"-")?T.vjb(e,0,w.Xzb(e).length-1)+"+":e))),D)?(D=new E.lz(D),U.Uvb(t.c,e,D),D):null));},E.qy=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M;t.b|=e;try{for(v=E.Dy("torsionID.txt"),P=(e&1)==0?null:E.Dy("torsionAngle.txt"),h=(e&1)==0?null:E.Dy("torsionRange.txt"),u=(e&1)==0?null:E.Dy("torsionFrequency.txt"),i=(e&2)==0?null:E.Dy("torsionBins.txt"),M=Rt.hgb(v);M!=null;){if(C=U.jpb(t.c,M),C||(C=new E.kz((c=w.Xzb("<").length,T.ejb(w.Xzb(M).substr(w.Xzb(M).length-c,c),"<")||(g=w.Xzb(">").length,T.ejb(w.Xzb(M).substr(w.Xzb(M).length-g,g),">"))?0:(d=w.Xzb("-").length,T.ejb(w.Xzb(M).substr(w.Xzb(M).length-d,d),"-")||(m=w.Xzb("+").length,T.ejb(w.Xzb(M).substr(w.Xzb(M).length-m,m),"+"))?1:(b=w.Xzb("=").length,T.ejb(w.Xzb(M).substr(w.Xzb(M).length-b,b),"=")?3:2)))),U.Uvb(t.c,M,C)),P)for(A=T.sjb(Rt.hgb(P),","),C.a=z(l.yeb,Kt,6,A.length,15,1),Q=0;Q<A.length;Q++)C.a[Q]=T.Qhb(A[Q],KA,32767)<<16>>16;if(h)for(s=T.sjb(Rt.hgb(h),","),C.d=h1(l.yeb,[P0,Kt],[16,6],15,[s.length,2],2),Q=0;Q<s.length;Q++)L=T.kjb(s[Q],45,1),C.d[Q][0]=T.Qhb(T.vjb(s[Q],0,L),KA,32767)<<16>>16,C.d[Q][1]=T.Qhb(T.ujb(s[Q],L+1),KA,32767)<<16>>16;if(u)for(o=T.sjb(Rt.hgb(u),","),C.c=z(l.yeb,Kt,6,o.length,15,1),Q=0;Q<o.length;Q++)C.c[Q]=T.Qhb(o[Q],-128,127)<<24>>24;if(i)for(r=T.sjb(Rt.hgb(i),","),C.b=z(l._6,J,6,r.length,15,1),Q=0;Q<r.length;Q++)C.b[Q]=T.Qhb(r[Q],-128,127)<<24>>24;M=Rt.hgb(v);}}catch(S){if(S=H1(S),Jt(S,76))D=S,T.LB(D,(T.Ujb(),T.Tjb),"");else throw Ut(S);}},E.ry=function(t){var e,A,P,r;for(t.a=15,r=(A=new U.Job(t.c).a.ad().Sb(),new U.Qob(A));r.a.cd();)P=(e=r.a.dd(),e.gd()),E.iz(P);},E.sy=function(){this.c=new U.awb(),this.b=0;},E.ty=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(P=z(l.b7,jt,6,2,15,1),b=E.wy(t,A),o=0;o<2;o++)if(A[3*o]!=-1){for(i=2-o,c=3-3*o,Q=0,h=0;h<3;h++)D=n.Yo(t,A[i],h),D!=b[1-o]&&t.C[D]!=1&&(A[c]=D,e?P[Q++]=E.Sx(e,A):P[Q++]=n.dl(t,A));return A[c]=-1,E.vy(P);}for(r=z(l.b7,jt,6,2,15,1),L=0,u=0;u<3;u++)if(g=n.Yo(t,A[1],u),g!=b[0]&&t.C[g]!=1){for(A[0]=g,s=0,h=0;h<3;h++)d=n.Yo(t,A[2],h),d!=b[1]&&t.C[d]!=1&&(A[3]=d,e?r[s++]=E.Sx(e,A):r[s++]=n.dl(t,A));P[L++]=E.vy(r);}return A[0]=-1,A[3]=-1,E.vy(P);},E.uy=function(t,e){return e[0]!=-1&&e[3]!=-1?E.Sx(t,e):E.ty(t.o,t,e);},E.vy=function(t){var e,A;return A=(t[1]+t[0])/2,e=j.Math.abs(t[1]-t[0]),e>n1?A:A<0?A+n1:A-n1;},E.wy=function(t,e){var A,P,r;return r=z(l.d7,H,6,2,15,1),n.Wo(t,e[1],e[2])!=-1?(r[0]=e[2],r[1]=e[1]):(A=z(l.d7,H,6,16,15,1),P=n.xp(t,A,e[1],e[2],15,null,null),r[0]=A[1],r[1]=A[P-1]),r;},E.xy=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;for(n.aw(t,7),Q=0,r=0;r<t.g;r++)n.em(t,r)==1&&!(r<t.g&&n.ot(t.p,r))&&n.sp(t,t.D[0][r])>1&&n.sp(t,t.D[1][r])>1&&(t.F[r]&64)==0&&!((t.F[r]&128)!=0&&(t.p&&r<t.g?n.et(t.p,r):0)<=5)&&(e[r]=true,++Q);for(D=z(l.d7,H,6,2,15,1),b=z(l.d7,H,6,2,15,1),i=null,P=0;P<t.g;P++)if(e[P]&&(i==null||!i[P])){for(A=0,s=0;s<2;s++)for(D[s]=t.D[s][P],b[s]=t.D[1-s][P],g=b[s];n.Po(t,D[s])==2&&n.sp(t,D[s])==2&&n.Sl(t,D[s])<10;){for(h=0;h<2;h++)if(u=n.Yo(t,D[s],h),u!=b[s]){o=n.$o(t,D[s],h),e[o]&&n.em(t,o)==1&&(e[o]=false,--Q),b[s]=D[s],D[s]=u,++A;break;}if(D[s]==g){A=0;break;}}A!=0&&(e[P]=false,--Q,n.sp(t,D[0])>1&&n.sp(t,D[1])>1&&(d=n.Kp(t,b[0],D[0]),m=n.Kp(t,b[1],D[1]),L=d<m?0:1,c=n.Wo(t,b[L],D[L]),i==null&&(i=z(l.zeb,ht,6,t.g,16,1)),i[c]=true,e[c]=true,++Q));}return Q;},E.yy=function(t){var e;return e=E.py(E.oy,t),e?E.fz(e):null;},E.zy=function(t,e,A,P){return E._y(P,t,e),A!=null&&(A[0]=P.f[0],A[1]=P.b[0],A[2]=P.b[1],A[3]=P.f[1]),P.d;},E.Ay=function(t){var e;return e=E.py(E.oy,t),e?E.gz(e):null;},E.By=function(t){var e;return e=E.py(E.oy,t),e?E.ez(e):null;},E.Cy=function(){var t;!E.oy&&(E.oy=new E.sy()),(E.oy.b&1)!=0&&E.oy.a!=15&&T.Ujb(),t=1&~E.oy.b,t!=0&&(E.qy(E.oy,t),t==1&&E.ry(E.oy));},E.Dy=function(t){var e,A,P;if(E.ny==null){if(e=_t.hAb("/resources/csd/"+t),e)return E.ny="csd/",new Rt.igb(new Rt.ngb((P=e,V1.Hmb(),P)));E.ny="cod/";}return new Rt.igb(new Rt.ngb((A=_t.hAb("/resources/"+E.ny+t),V1.Hmb(),A)));},x(365,1,{},E.sy),y.a=0,y.b=0,l.l8=X(365),E.Ey=function(t,e){var A,P;for(P=0;P<t.b.length;P++)if(A=j.Math.abs(t.b[P]-e.b[P]),A>lP&&A<t.a[P]-lP)return A<t.a[P]/2^t.b[P]<e.b[P]?1:-1;return 0;},E.Fy=function(t,e){var A,P;for(P=0;P<t.b.length;P++)if(A=j.Math.abs(t.b[P]-e.b[P]),A>lP&&A<t.a[P]-lP)return  false;return  true;},E.Gy=function(t,e){this.b=t,this.a=e;},x(256,1,{256:1,32:1},E.Gy),y.Rb=function(t){return E.Ey(this,t);},y.ob=function(){var t,e;for(e=new T.Pjb(),t=0;t<this.b.length;t++)e.a+=t==0?"Torsions: ":", ",T.Njb(e,(lt.Q3(),lt.R3(this.b[t],3)+"("+Ft(j.Math.round(this.a[t]+.5))+")"));return e.a;},l.n8=X(256),E.Ly=function(){E.Ly=o1,E.Ky=F(k(l.b7,1),jt,6,15,[X1,n1,En,b0]),E.Jy=F(k(l.d7,2),S1,7,0,[F(k(l.d7,1),H,6,15,[0,0,0,2]),F(k(l.d7,1),H,6,15,[0,0,1,2]),F(k(l.d7,1),H,6,15,[0,1,1,3]),F(k(l.d7,1),H,6,15,[2,2,3,2])]);},E.My=function(t,e,A,P){var r,i,D,u;if(P==1&&!E.Ry(t,e)){if(n.Zo(t.b,e)==3)return  -1;if(n.Zo(t.b,e)==3)return r=E.Py(t,e,A),n.lw(t.b,r[0])==n.lw(t.b,r[1])?r[2]:n.lw(t.b,r[0])==n.lw(t.b,r[2])?r[1]:r[0];}for(D=-1,u=-1,i=0;i<n.Zo(t.b,e);i++)r=n.Yo(t.b,e,i),r!=A&&D<n.lw(t.b,r)&&(D=n.lw(t.b,r),u=r);return u;},E.Ny=function(t){var e,A,P,r,i;for(t.a=h1(l.d7,[S1,H],[7,6],15,[t.d.length,4],2),t.c=h1(l.d7,[S1,H],[7,6],15,[t.d.length,2],2),t.e=z(l.d7,H,6,t.d.length,15,1),e=z(l.d7,H,6,2,15,1),r=0;r<t.d.length;r++){for(i=0;i<2;i++)e[0]=n.Yl(t.b,1-i,t.d[r]),e[1]=n.Yl(t.b,i,t.d[r]),E.Xy(t.b,e[1])&&E.Vy(t.b,e[0],e,null),t.a[r][1+i]=e[1],t.c[r][i]=e[0];A=E.Oy(t,t.a[r][1],t.c[r][0]),P=E.Oy(t,t.a[r][2],t.c[r][1]),t.e[r]=E.Jy[A][P],t.a[r][0]=E.My(t,t.a[r][1],t.c[r][0],A),t.a[r][3]=E.My(t,t.a[r][2],t.c[r][1],P);}},E.Oy=function(t,e,A){var P;if(n.Zo(t.b,e)==2)return 1;if(P=E.Py(t,e,A),n.Zo(t.b,e)==3)return n.lw(t.b,P[0])==n.lw(t.b,P[1])?E.Ry(t,e)?2:1:E.Ry(t,e)?1:0;if(n.Zo(t.b,e)==4){if(n.lw(t.b,P[0])==n.lw(t.b,P[1])&&n.lw(t.b,P[0])==n.lw(t.b,P[2]))return 3;if(n.lw(t.b,P[0])==n.lw(t.b,P[1])||n.lw(t.b,P[0])==n.lw(t.b,P[2])||n.lw(t.b,P[1])==n.lw(t.b,P[2]))return 1;}return 0;},E.Py=function(t,e,A){var P,r,i;for(i=0,P=z(l.d7,H,6,n.Zo(t.b,e)-1,15,1),r=0;r<n.Zo(t.b,e);r++)n.Yo(t.b,e,r)!=A&&(P[i++]=n.Yo(t.b,e,r));return P;},E.Qy=function(t,e){var A,P,r;for(r=z(l.c7,w0,6,t.d.length,15,1),P=z(l.c7,w0,6,t.d.length,15,1),A=0;A<t.d.length;A++)r[A]=E.Wy(E.uy(e,t.a[A]),t.e[A]),P[A]=E.Ky[t.e[A]];return new E.Gy(r,P);},E.Ry=function(t,e){return !!(n.Po(t.b,e)==1&&n.Sl(t.b,e)<10||n.Sp(t.b,e)||n.Zp(t.b,e,true));},E.Sy=function(t){E.Ly(),this.b=t,n.aw(this.b,63),this.d=E.Uy(t),E.Ny(this);},E.Ty=function(t,e){E.Ly(),this.b=t,n.aw(this.b,63),this.d=e,E.Ny(this);},E.Uy=function(t){E.Ly();var e,A,P,r,i;for(n.aw(t,7),r=z(l.zeb,ht,6,t.g,16,1),P=0,A=0;A<t.g;A++)E.Yy(t,A)&&(r[A]=true,++P);for(i=z(l.d7,H,6,P,15,1),P=0,e=0;e<t.g;e++)r[e]&&(i[P++]=e);return i;},E.Vy=function(t,e,A,P){var r,i,D;for(r=0;r<2;r++)if(i=n.Yo(t,A[1],r),i!=A[0])return i==e||(D=n.$o(t,A[1],r),A[0]=A[1],A[1]=i,t.j[i]==1)?false:(P!=null&&!n.Jm(t,A[0])&&!n.Jm(t,A[1])&&(P[0]=j.Math.max(P[0],D)),t.j[i]==2&&t.o[i]==2&&t.C[i]<=7?E.Vy(t,e,A,P):true);return  false;},E.Wy=function(t,e){var A;for(A=E.Ky[e]/2;t<-A;)t+=X1;for(;t>=A;)t-=E.Ky[e];return t;},E.Xy=function(t,e){return t.j[e]==2&&t.o[e]==2&&t.C[e]<=7;},E.Yy=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;if(n.em(t,e)!=1||e<t.g&&n.ot(t.p,e)||(t.p&&e<t.g?n.et(t.p,e):0)==3)return  false;for(A=z(l.d7,H,6,2,15,1),u=0;u<2;u++)if(A[u]=t.D[u][e],n.Jm(t,A[u])||n.sp(t,A[u])<=1)return  false;if(E.Xy(t,A[0])||E.Xy(t,A[1])){for(s=z(l.d7,H,6,1,15,1),s[0]=e,P=z(l.d7,S1,7,2,0,2),o=0;o<2;o++)if(E.Xy(t,A[o])&&(P[o]=z(l.d7,H,6,2,15,1),P[o][0]=A[1-o],P[o][1]=A[o],!E.Vy(t,A[1-o],P[o],s)))return  false;if(e!=s[0])return  false;for(Q=0;Q<2;Q++)P[Q]!=null&&(A[Q]=P[Q][1]);}for(D=0;D<2;D++){if(i=n.Zo(t,A[D]),i==1)return  false;for(h=0,L=0;L<i;L++)r=n.Yo(t,A[D],L),(t.u[r]&A0)!=0||++h;if(h<2)return  false;}return  true;},x(254,1,{},E.Sy,E.Ty),l.m8=X(254),E.$y=function(){E.$y=o1,E.Zy=F(k(l.d7,2),S1,7,0,[F(k(l.d7,1),H,6,15,[0,0,1]),F(k(l.d7,1),H,6,15,[0,2,3]),F(k(l.d7,1),H,6,15,[1,3,3])]);},E._y=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt;if(n.$v(t.c),t.d=null,n.aw(e,63),n.em(e,A)!=1||A<e.g&&n.ot(e.p,A)||n.Sl(e,e.D[0][A])==1||n.Sl(e,e.D[1][A])==1||(Lt=(e.F[A]&128)!=0,Lt&&(e.p&&A<e.g?n.et(e.p,A):0)<6))return  false;for(r=z(l.zeb,ht,6,e.f,16,1),t.a=0,O=0;O<2;O++){for(t.b[O]=e.D[O][A],t.e[O]=e.D[1-O][A];n.Po(e,t.b[O])==2&&n.sp(e,t.b[O])==2&&n.Sl(e,t.b[O])<10;)for(ot=0;ot<n.Zo(e,t.b[O]);ot++)if(D=n.Yo(e,t.b[O],ot),D!=t.e[O]&&e.C[D]!=1){if(e.j[D]==1||t.a==16)return  false;r[t.b[O]]=true,t.e[O]=t.b[O],t.b[O]=D,++t.a;break;}if(rt=n.sp(e,t.b[O]),rt>4||rt==1)return  false;r[t.b[O]]=true;}for(K=0;K<2;K++)for(ot=0;ot<n.Zo(e,t.b[K]);ot++)D=n.Yo(e,t.b[K],ot),e.C[D]!=1&&(r[D]=true);for(t.g=z(l.d7,H,6,e.f,15,1),n.Bo(e,t.c,r,true,t.g),N=0;N<t.g.length;N++)t.g[N]!=-1&&(t.i[t.g[N]]=N);if(n.Nn(t.c,true),Lt&&(i=n.Wo(t.c,t.g[t.b[0]],t.g[t.b[1]]),i!=-1)){for(n.Kn(t.c,i,256,true),vt=(n.aw(e,7),e.p),$t=0;$t<vt.j.a.length;$t++)if(n.qt(vt,$t,A)){for(I=0;I<2;I++)for(ot=0;ot<n.Zo(e,t.b[I]);ot++)if(D=n.Yo(e,t.b[I],ot),D!=t.e[I]&&n.pt(vt,$t,D)&&e.C[D]!=1){n.Kn(t.c,n.Wo(t.c,t.g[t.b[I]],t.g[D]),256,true);break;}}}for(ut=0;ut<2;ut++){for(n.Zp(e,t.b[ut],true)&&n.un(t.c,t.g[t.b[ut]],Ie,true),o=false,ot=0;ot<n.Zo(e,t.b[ut]);ot++)D=n.Yo(e,t.b[ut],ot),D!=t.e[ut]&&e.C[D]!=1&&(m=n.Wo(t.c,t.g[t.b[ut]],t.g[D]),n.hm(t.c,m)==8?o=true:e.C[D]==6&&!n.Sp(e,t.b[ut])&&(d=D<e.f&&n.nt(e.p,D)?2:4,n.un(t.c,t.g[D],d,true)),u=n.$o(e,t.b[ut],ot),Ht=e.p&&u<e.g?n.et(e.p,u):0,(Ht==3||Ht==4)&&n.Kn(t.c,m,Ht<<17,true),(u<e.g&&n.ot(e.p,u)||n._o(e,t.b[ut],ot)==2)&&(rt=n.sp(e,D),S=rt==3,!S&&rt==2&&(e.u[D]&V0)==0&&(S=n.Lp(e,t.b[1-ut],u)!=-1),S?n.un(t.c,t.g[D],Rn,true):u<e.g&&n.ot(e.p,u)&&n.un(t.c,t.g[D],pn,true)),n._o(e,t.b[ut],ot)==1&&(n.sp(e,D)==4?n.un(t.c,t.g[D],Qn,true):e.C[D]==6&&n.un(t.c,t.g[D],F0,true)));o||(n.Sp(e,t.b[ut])?n.un(t.c,t.g[t.b[ut]],2,true):n.un(t.c,t.g[t.b[ut]],4,true));}for(n.aw(t.c,191),V=0;V<2;V++)if(C=t.g[t.b[V]],n.Ml(t.c,C)==3){if(n.Ml(e,t.b[V])==3)return  false;yt=n.Qo(t.c,C),n.Ln(t.c,yt,257),n.Yl(t.c,0,yt)!=C&&(n.Cn(t.c,1,yt,n.Yl(t.c,0,yt)),n.Cn(t.c,0,yt,C)),n.aw(t.c,191);}if(h=t.g[t.b[0]],b=t.g[t.b[1]],ct=t.g[t.e[0]],wt=t.g[t.e[1]],Q=n.Gl(t.c,h),L=n.Gl(t.c,b),n.ym(t.c,h)&&n.ym(t.c,b)&&(Q!=0||L!=0)&&(Q!=L||n.Fl(t.c,h)!=n.Fl(t.c,b)))return  false;if(s=false,n.ym(t.c,h)&&Q!=0&&(n.nn(t.c,h,0,-1),s=true),n.ym(t.c,b)&&L!=0&&(n.nn(t.c,b,0,-1),s=true),s&&n.aw(t.c,191),bt=n.lw(t.c,h),it=n.lw(t.c,b),bt<it&&(Nt=t.b[0],t.b[0]=t.b[1],t.b[1]=Nt,Nt=t.e[0],t.e[0]=t.e[1],t.e[1]=Nt,Nt=h,h=b,b=Nt,Nt=ct,ct=wt,wt=Nt),Dt=false,(n.ym(t.c,h)||n.ym(t.c,b))&&(n.ym(t.c,h)?Dt=n.bw(t.c,h)==1:n.ym(t.c,b)&&(Dt=n.bw(t.c,b)==1),Dt)){for(P=0;P<t.c.q;P++)n.yn(t.c,P,-n.Pl(t.c,P));n.aw(t.c,191);}return c=E.bz(t,h,ct),g=E.bz(t,b,wt),t.f[0]=c==-1?-1:t.i[c],t.f[1]=g==-1?-1:t.i[g],Qt=n.iw(t.c),Qt==null?false:(v=E.az(t,h,ct),M=E.az(t,b,wt),v==0&&M==0&&(t.c.I&fe)==A0?Ot=2:Ot=E.Zy[v][M],xt=Ot==0?Dt?"<":">":Ot==1?Dt?"-":"+":Ot==3?"=":"",t.d=Qt+xt,true);},E.az=function(t,e,A){var P,r,i,D;if(n.Zo(t.c,e)==2)return 1;if(P=E.cz(t,e,A),n.Zo(t.c,e)==3)return n.lw(t.c,P[0])==n.lw(t.c,P[1])?n.Po(t.c,e)==1&&n.Sl(t.c,e)<10||n.Sp(t.c,e)||n.Zp(t.c,e,true)?2:1:n.Po(t.c,e)==1&&n.Sl(t.c,e)<10||n.Sp(t.c,e)||n.Zp(t.c,e,true)?1:0;if(n.Zo(t.c,e)==4){for(r=0;r<P.length;r++)for(D=n.lw(t.c,P[r]),i=r+1;i<P.length;i++)if(D==n.lw(t.c,P[i]))return 1;}return 0;},E.bz=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h;for(L=-1,s=-1,h=-1,i=z(l.zeb,ht,6,n.Zo(t.c,e),16,1),o=0;o<n.Zo(t.c,e);o++)if(!i[o]&&(r=n.Yo(t.c,e,o),r!=A&&(D=n.lw(t.c,r),s<D))){for(u=false,Q=o+1;Q<n.Zo(t.c,e);Q++)if(P=n.Yo(t.c,e,Q),P!=A&&n.lw(t.c,P)==D){if(i[Q]=true,u)return r;u=true;}u?h=r:(s=D,L=r);}return L==-1&&(n.Po(t.c,e)==1&&n.Sl(t.c,e)<10||n.Sp(t.c,e)||n.Zp(t.c,e,true))?h:L;},E.cz=function(t,e,A){var P,r,i;for(i=0,P=z(l.d7,H,6,n.Zo(t.c,e)-1,15,1),r=0;r<n.Zo(t.c,e);r++)n.Yo(t.c,e,r)!=A&&(P[i++]=n.Yo(t.c,e,r));return P;},E.dz=function(){E.$y(),this.c=new n.tw(24,29),this.i=z(l.d7,H,6,24,15,1),this.b=z(l.d7,H,6,2,15,1),this.e=z(l.d7,H,6,2,15,1),this.f=z(l.d7,H,6,2,15,1);},x(250,1,{},E.dz),y.a=0,l.o8=X(250),E.ez=function(t){var e,A,P,r,i,D,u,o,Q;switch(e=null,t.e){case 1:for(e=z(l.yeb,Kt,6,2*t.a.length,15,1),P=0;P<t.a.length;P++)e[P]=t.a[P],e[t.a.length+P]=180+t.a[P]<<16>>16;return e;case 2:for(r=t.a[0]==0?1:0,i=t.a[t.a.length-1]==180?t.a.length-1:t.a.length,Q=i-r,e=z(l.yeb,Kt,6,t.a.length+Q,15,1),D=0;D<t.a.length;D++)e[D]=t.a[D];for(u=r;u<i;u++)e[e.length-1-u+r]=360-t.a[u]<<16>>16;return e;case 3:for(r=t.a[0]==0?1:0,i=t.a[t.a.length-1]==90?t.a.length-1:t.a.length,Q=i-r,e=z(l.yeb,Kt,6,2*t.a.length+2*Q,15,1),o=0;o<t.a.length;o++)e[o]=t.a[o],e[t.a.length+Q+o]=180+t.a[o]<<16>>16;for(A=r;A<i;A++)e[t.a.length+Q-1-A+r]=180-t.a[A]<<16>>16,e[e.length-1-A+r]=360-t.a[A]<<16>>16;return e;default:return t.a;}},E.fz=function(t){var e,A,P,r,i,D,u,o,Q,L;switch(e=null,L=t.c.length,t.e){case 1:for(e=z(l.yeb,Kt,6,2*L,15,1),P=0;P<L;P++)e[P]=t.c[P],e[L+P]=t.c[P];return e;case 2:for(r=t.a[0]==0?1:0,i=t.a[L-1]==180?L-1:L,Q=i-r,e=z(l.yeb,Kt,6,L+Q,15,1),D=0;D<L;D++)e[D]=t.c[D];for(u=r;u<i;u++)e[e.length-1-u+r]=t.c[u];return e;case 3:for(r=t.a[0]==0?1:0,i=t.a[L-1]==90?L-1:L,Q=i-r,e=z(l.yeb,Kt,6,2*L+2*Q,15,1),o=0;o<L;o++)e[o]=t.c[o],e[L+Q+o]=t.c[o];for(A=r;A<i;A++)e[L+Q-1-A+r]=t.c[A],e[e.length-1-A+r]=t.c[A];return e;default:return t.c;}},E.gz=function(t){var e,A,P,r,i,D,u,o,Q,L;switch(e=null,L=t.d.length,t.e){case 1:for(e=h1(l.yeb,[P0,Kt],[16,6],15,[2*L,2],2),P=0;P<L;P++)e[P][0]=t.d[P][0],e[P][1]=t.d[P][1],e[L+P][0]=180+t.d[P][0]<<16>>16,e[L+P][1]=180+t.d[P][1]<<16>>16;return e;case 2:for(r=t.a[0]==0?1:0,i=t.a[L-1]==180?L-1:L,Q=i-r,e=h1(l.yeb,[P0,Kt],[16,6],15,[L+Q,2],2),D=0;D<L;D++)e[D][0]=t.d[D][0],e[D][1]=t.d[D][1];for(u=r;u<i;u++)e[e.length-1-u+r][0]=360-t.d[u][1]<<16>>16,e[e.length-1-u+r][1]=360-t.d[u][0]<<16>>16;return e;case 3:for(r=t.a[0]==0?1:0,i=t.a[L-1]==90?L-1:L,Q=i-r,e=h1(l.yeb,[P0,Kt],[16,6],15,[2*L+2*Q,2],2),o=0;o<L;o++)e[o][0]=t.d[o][0],e[o][1]=t.d[o][1],e[L+Q+o][0]=180+t.d[o][0]<<16>>16,e[L+Q+o][1]=180+t.d[o][1]<<16>>16;for(A=r;A<i;A++)e[L+Q-1-A+r][0]=180-t.d[A][1]<<16>>16,e[L+Q-1-A+r][1]=180-t.d[A][0]<<16>>16,e[e.length-1-A+r][0]=360-t.d[A][1]<<16>>16,e[e.length-1-A+r][1]=360-t.d[A][0]<<16>>16;return e;default:return t.d;}},E.hz=function(t,e,A,P,r,i){var D,u,o,Q,L,s;for(t.a[A]=e<<16>>16,t.c[A]=t.c[A]+t.c[P]<<16>>16,t.d[A][0]=r<<16>>16,t.d[A][1]=i<<16>>16,s=t.a.length-1,o=z(l.yeb,Kt,6,s,15,1),Q=z(l.yeb,Kt,6,s,15,1),L=z(l.yeb,P0,16,s,0,2),u=0,D=0;D<t.a.length;D++)D!=P&&(o[u]=t.a[D],Q[u]=t.c[D],L[u]=t.d[D],++u);t.a=o,t.c=Q,t.d=L;},E.iz=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h;for(e=0;t.a.length!=0;){if(t.a.length==1){if(t.e==1||t.e==0)break;if(t.e==2&&(t.a[0]==0||t.a[0]==180))break;if(t.e==3&&(t.a[0]==0||t.a[0]==90))break;}for(h=t.a[0]!=0&&(t.e==2||t.e==3)?-1:0,s=-1,L=St,D=h;D<t.a.length;D++)o=D+1,A=D>=0?t.a[D]:-t.a[0],P=o<t.a.length?t.a[o]:t.e==1?t.a[0]+180:t.e==2?360-t.a[t.a[D]==180?D-1:D]:t.e==3?180-t.a[t.a[D]==90?D-1:D]:t.a[0]+360,L>P-A&&(L=P-A,s=D);if(L>15)break;i=s,u=i+1,i==-1?(e+=t.c[0],t.a[0]=0,t.c[0]=t.c[0]*2<<16>>16,t.d[0][0]=-t.d[0][1]):u<t.a.length?(Q=E.jz(t,i,u,t.a[u]),E.hz(t,Q,i,u,t.d[i][0],t.d[u][1])):t.e==1?(u=0,Q=E.jz(t,i,u,t.a[u]+180),Q<=180?E.hz(t,Q,i,u,t.d[i][0],t.d[u][1]+180):E.hz(t,Q-180,u,i,t.d[u][0],t.d[i][1]-180)):t.e==2?(u=t.a[i]==180?i-1:i,t.a[i]<180?(e+=t.c[i],t.a[i]=180,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=360-t.d[i][0]<<16>>16):(T.Ujb(),Q=E.jz(t,i,u,t.a[u]),E.hz(t,Q,u,i,t.d[u][0],t.d[i][1]))):t.e==3?(u=t.a[i]==90?i-1:i,t.a[i]<90?(e+=t.c[i],t.a[i]=90,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=180-t.d[i][0]<<16>>16):(T.Ujb(),Q=E.jz(t,i,u,t.a[u]),E.hz(t,Q,u,i,t.d[u][0],t.d[i][1]))):(u=0,Q=E.jz(t,i,u,t.a[u]+360),Q<=360?E.hz(t,Q,i,u,t.d[i][0],t.d[u][1]+360):E.hz(t,Q-360,u,i,t.d[u][0],t.d[i][1]-360));}if(e!=0)for(r=0;r<t.c.length;r++)t.c[r]=Ft(j.Math.round(100*t.c[r]/(100+e)))<<16>>16;},E.jz=function(t,e,A,P){return Ft(j.Math.round((t.a[e]*t.c[e]+P*t.c[A])/(t.c[e]+t.c[A])));},E.kz=function(t){this.e=t;},E.lz=function(t){var e;if(this.e=t.e,t.a!=null)for(this.a=z(l.yeb,Kt,6,t.a.length,15,1),e=0;e<this.a.length;e++)this.a[e]=360-t.a[this.a.length-e-1]<<16>>16;if(t.d!=null)for(this.d=h1(l.yeb,[P0,Kt],[16,6],15,[t.d.length,2],2),e=0;e<this.d.length;e++)this.d[e][0]=360-t.d[this.a.length-e-1][1]<<16>>16,this.d[e][1]=360-t.d[this.a.length-e-1][0]<<16>>16;if(t.c!=null)for(this.c=z(l.yeb,Kt,6,t.c.length,15,1),e=0;e<this.c.length;e++)this.c[e]=t.c[this.c.length-e-1];if(t.b!=null)for(this.b=z(l._6,J,6,t.b.length,15,1),e=0;e<this.b.length;e++)this.b[e]=t.b[this.b.length-e-1];},x(194,1,{194:1},E.kz,E.lz),y.e=0,l.p8=X(194),E.mz=function(t,e){var A,P,r,i;A=e[1],P=e[2],r=t.j[A]-1,i=t.j[P]-1,r>3||i>3?(this.b=z(l.yeb,Kt,6,4,15,1),this.b[0]=45,this.b[1]=135,this.b[2]=225,this.b[2]=315,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[4,2],2),this.c[0][0]=30,this.c[0][1]=60,this.c[1][0]=120,this.c[1][1]=150,this.c[2][0]=210,this.c[2][1]=240,this.c[2][0]=300,this.c[2][1]=330,this.a=z(l.yeb,Kt,6,4,15,1),this.a[0]=25,this.a[1]=25,this.a[2]=25,this.a[3]=25):(t.o[A]==0||t.C[A]>9)&&(t.o[P]==0||t.C[P]>9)?r==3&&i==3||r==3&&i==2||r==3&&i==1||r==2&&i==3||r==1&&i==3||r==2&&i==2&&(e[0]!=-1||e[3]!=-1)?(this.b=z(l.yeb,Kt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=z(l.yeb,Kt,6,3,15,1),this.a[0]=33,this.a[1]=33,this.a[2]=33):r==1&&i==2&&e[3]==-1||r==2&&i==1&&e[0]==-1?(this.b=z(l.yeb,Kt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=z(l.yeb,Kt,6,3,15,1),this.a[0]=40,this.a[1]=20,this.a[2]=40):(r==1&&i==1||r==1&&i==2&&e[3]!=-1||r==2&&i==1&&e[0]!=-1||r==2&&i==2&&e[0]==-1&&e[3]==-1)&&(this.b=z(l.yeb,Kt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=z(l.yeb,Kt,6,3,15,1),this.a[0]=25,this.a[1]=50,this.a[2]=25):(t.o[A]==0||t.C[A]>9)&&t.o[P]==1||(t.o[P]==0||t.C[P]>9)&&t.o[A]==1?r==3||i==3?(this.b=z(l.yeb,Kt,6,6,15,1),this.b[0]=0,this.b[1]=60,this.b[2]=120,this.b[3]=180,this.b[4]=240,this.b[5]=300,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=45,this.c[1][1]=75,this.c[2][0]=105,this.c[2][1]=135,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=225,this.c[4][1]=255,this.c[5][0]=285,this.c[5][1]=315,this.a=z(l.yeb,Kt,6,6,15,1),this.a[0]=16,this.a[1]=16,this.a[2]=16,this.a[3]=16,this.a[4]=16,this.a[5]=16):r==1&&i==1?(this.b=z(l.yeb,Kt,6,2,15,1),this.b[0]=120,this.b[1]=240,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[2,2],2),this.c[0][0]=105,this.c[0][1]=135,this.c[1][0]=225,this.c[1][1]=255,this.a=z(l.yeb,Kt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[A]==1&&r==2&&i==1||t.o[P]==1&&i==2&&r==1?(this.b=z(l.yeb,Kt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=z(l.yeb,Kt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[A]==1&&r==1&&i==2&&e[3]==-1||t.o[P]==1&&i==1&&r==2&&e[0]==-1?(this.b=z(l.yeb,Kt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=z(l.yeb,Kt,6,3,15,1),this.a[0]=60,this.a[1]=20,this.a[2]=20):t.o[A]==1&&r==1&&i==2&&e[3]!=-1||t.o[P]==1&&i==1&&r==2&&e[0]!=-1?(this.b=z(l.yeb,Kt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=z(l.yeb,Kt,6,3,15,1),this.a[0]=20,this.a[1]=40,this.a[2]=40):r==2&&i==2&&(e[0]==-1||e[3]==-1?(this.b=z(l.yeb,Kt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=z(l.yeb,Kt,6,2,15,1),this.a[0]=50,this.a[1]=50):(this.b=z(l.yeb,Kt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=z(l.yeb,Kt,6,2,15,1),this.a[0]=50,this.a[1]=50)):t.o[A]==1&&t.o[P]==1?r==1&&i==1?(this.b=z(l.yeb,Kt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=z(l.yeb,Kt,6,2,15,1),this.a[0]=10,this.a[1]=90):(this.b=z(l.yeb,Kt,6,6,15,1),this.b[0]=0,this.b[1]=50,this.b[2]=130,this.b[3]=180,this.b[4]=230,this.b[5]=310,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=35,this.c[1][1]=65,this.c[2][0]=115,this.c[2][1]=145,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=215,this.c[4][1]=245,this.c[5][0]=295,this.c[5][1]=325,this.a=z(l.yeb,Kt,6,6,15,1),this.a[0]=40,this.a[1]=5,this.a[2]=5,this.a[3]=40,this.a[4]=5,this.a[5]=5):(this.b=z(l.yeb,Kt,6,1,15,1),this.b[0]=180,this.c=h1(l.yeb,[P0,Kt],[16,6],15,[1,2],2),this.c[0][0]=165,this.c[0][1]=195,this.a=z(l.yeb,Kt,6,1,15,1),this.a[0]=100);},x(251,1,{},E.mz),l.q8=X(251),E.pz=function(){E.pz=o1,E.oz=F(k(l.c7,1),w0,6,15,[1,hA,dr,2.119999885559082,1.9800000190734863,1.909999966621399,1.7699999809265137,QP,1.5,jr,1.5800000429153442,2.5,2.509999990463257,2.25,2.190000057220459,Br,1.8899999856948853,h9,1.8300000429153442,2.7300000190734863,2.619999885559082,2.5799999237060547,T9,b9,LP,LP,mr,c9,c9,2.380000114440918,2.390000104904175,kr,Fr,1.8799999952316284,h9,w9,2.25,3.2100000381469727,2.8399999141693115,2.75,2.5199999809265137,2.559999942779541,LP,mr,T9,mr,2.1500000953674316,g9,z9,2.430000066757202,b9,d9,1.9900000095367432,2.0399999618530273,2.059999942779541,3.4800000190734863,3.0299999713897705,2.9800000190734863,j9,B9,2.950000047683716,B9,2.9000000953674316,m9,vr,k9,m9,F9,vr,k9,v9,2.740000009536743,2.630000114440918,g9,2.569999933242798,z9,2.4800000190734863,2.4100000858306885,Fr,kr,LP,d9,2.5999999046325684,2.5399999618530273,2.5,2.5,2.5,2.5,2.5,v9,2.930000066757202,j9,2.7100000381469727,2.819999933242798,F9,vr,p9,3.4000000953674316,p9,2.700000047683716]),E.nz=F(k(l.c7,1),w0,6,15,[.25,pr,.46000000834465027,R9,1.0199999809265137,TA,.75,.7099999785423279,.6299999952316284,.6399999856948853,.9599999785423279,C9,H9,1.2599999904632568,Rr,M9,1.0299999713897705,.9900000095367432,S9,q9,V9,1.4800000190734863,Cr,E9,N9,1.190000057220459,Rr,M9,1.100000023841858,hA,hA,I9,yP,yP,Rr,1.1399999856948853,yP,x9,Hr,Mr,1.5399999618530273,O9,1.3799999952316284,X9,1.25,1.25,hA,H9,G9,jr,UA,UA,Cr,R9,1.350000023841858,kr,q9,oP,Mr,K9,1.7400000095367432,Y9,J9,UP,Z9,UP,_9,QP,Sr,1.6399999856948853,bA,W9,1.5199999809265137,jr,$9,1.309999942779541,a9,N9,1.2300000190734863,I9,1.4199999570846558,1.5,G9,1.5099999904632568,ti,O9,ti,2.2300000190734863,2.009999990463257,w9,1.75,Z9,bA,V9,J9,QP,QP,UP,UP,Sr,_9,Y9,K9,1.6100000143051147,ei,1.4900000095367432,dr,1.409999966621399,E9,a9,X9,yP,$9,Cr,dr,W9,1.75,Sr,ei]);},Yt.rz=function(){Yt.rz=o1,Yt.qz=n.oo("X",32);},Yt.sz=function(t,e){Yt.rz();var A,P,r,i,D;for(r=0;r<t.q;r++)for(D=n.ew(t),n.Bn(D,r,Yt.qz),n.aw(D,15),P=0;P<D.f;P++)(D.u[P]&q0)!=0&&n.Hp(D,P)==-1&&(i=(n.aw(D,7),D.o[P]==2&&D.j[P]==2?n.mq(D,P,false):n.oq(D,P,false)),i!=-1&&(t.H[i]=257,t.T=0,t.D[1][i]==P&&(A=t.D[0][i],t.D[0][i]=P,t.T=0,t.D[1][i]=A,t.T=0),n.nn(t,P,e,0)));},Yt.tz=function(t){Yt.rz();var e,A,P,r;for(Yt.sz(t,1),P=t.q,A=z(l.Sbb,st,2,P,6,1),e=0;e<P;e++)r=n.ew(t),n.ln(r,e,(n.Qk(),n.Kk)[r.C[e]]+"*"),r.C[e]==1?n.Bn(r,e,Yt.qz):n.sn(r,e,r.A[e]+5),Yt.uz(r),A[e]=n.qg(new n.Ng(r,8));return A;},Yt.uz=function(t){var e;for(n.aw(t,15),e=0;e<t.q;e++)(t.u[e]&3)!=0&&n.nn(t,e,1,0);},Yt.qz=0,Yt.vz=function(t,e,A){var P,r;for(P=h1(l.Sbb,[c0,st],[13,2],6,[t.f,e],2),n.aw(t,7),r=0;r<t.f;r++)P[r]=Yt.wz(t,r,e,A);return P;},Yt.wz=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g;for(o=new n.tw(t.f,t.g),L=new U.cxb(),b=0,h=0,D=z(l.zeb,ht,6,t.f,16,1),i=z(l.d7,H,6,t.f,15,1),g=0;g<A&&h<t.f;g++){if(h==0)i[0]=e,D[e]=true,h=1;else {for(c=h,Q=b;Q<h;Q++)for(r=i[Q],s=0;s<t.j[r];s++)if(u=t.i[r][s],!D[u])switch(P){case 0:D[u]=true,i[c++]=u;break;case 1:Yt.yz(t,r)&&Yt.yz(t,u)||(D[u]=true,i[c++]=u);}b=h,h=c;}n.Bo(t,o,D,true,null),U._wb(L,n.qg(new n.Ng(o,8)));}return U.bxb(L,z(l.Sbb,st,2,L.a.a.length,6,1));},Yt.xz=function(t,e,A){var P,r,i,D,u,o;for(u=n.Oq(new n.$q(true),t),r=-1,i=0;i<u.q;i++)if(P=u.t==null||u.t[i]==null?null:T.cjb((D=u.t[i],V1.Hmb(),D)),P!=null&&(o=w.Xzb("*").length,T.ejb(w.Xzb(P).substr(w.Xzb(P).length-o,o),"*"))){r=i;break;}return r>=0?Yt.wz(u,r,e,A):z(l.Sbb,st,2,0,6,1);},Yt.yz=function(t,e){return !(t.C[e]!=6||t.s[e]!=0||n.np(t,e)+t.j[e]!=4);},Yt.Jz=function(){Yt.Jz=o1,Yt.zz=j.Math.cos(Hn),Yt.Ez=j.Math.sin(Hn),Yt.Bz=j.Math.cos(qr),Yt.Gz=j.Math.sin(qr),Yt.Dz=j.Math.cos(b0),Yt.Iz=j.Math.sin(b0),Yt.Az=j.Math.cos(cA),Yt.Fz=j.Math.sin(cA),Yt.Cz=j.Math.cos(Qr),Yt.Hz=j.Math.sin(Qr),j.Math.cos(ni),j.Math.sin(ni);},Yt.Kz=function(t){Yt.Jz();var e,A;for(n.aw(t,1),A=t.f,e=0;e<A;e++)Yt.Lz(t,e);},Yt.Lz=function(t,e){Yt.Jz();var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot;switch(O=n.np(t,e),Lt=t.J[e].a,ot=t.J[e].b,O){case 1:{if(N=t.j[e],N==0?(L=n.Uo(t,true),b=0):(L=Lt-n.Pl(t,t.i[e][0]),b=ot-n.Ql(t,t.i[e][0])),N==1)v=n.Sk(t,Lt+Yt.Cz*L+Yt.Hz*b,ot-Yt.Hz*L+Yt.Cz*b,0);else if(N==2)L=Lt-.5*(n.Pl(t,t.i[e][0])+n.Pl(t,t.i[e][1])),b=ot-.5*(n.Ql(t,t.i[e][0])+n.Ql(t,t.i[e][1])),v=n.Sk(t,Lt+L,ot+b,0);else if(N==3){for(ut=t.i[e][0],m=1;m<3;m++)u=t.k[e][m],(t.H[u]==129||t.H[u]==257)&&(ut=t.i[e][m]);if(A=j.Math.abs(n.no(n.mo(t.J[e].a,t.J[e].b,n.Pl(t,t.i[e][0]),n.Ql(t,t.i[e][0])),n.mo(t.J[e].a,t.J[e].b,n.Pl(t,t.i[e][1]),n.Ql(t,t.i[e][1])))),P=j.Math.abs(n.no(n.mo(t.J[e].a,t.J[e].b,n.Pl(t,t.i[e][0]),n.Ql(t,t.i[e][0])),n.mo(t.J[e].a,t.J[e].b,n.Pl(t,t.i[e][2]),n.Ql(t,t.i[e][2])))),r=j.Math.abs(n.no(n.mo(t.J[e].a,t.J[e].b,n.Pl(t,t.i[e][1]),n.Ql(t,t.i[e][1])),n.mo(t.J[e].a,t.J[e].b,n.Pl(t,t.i[e][2]),n.Ql(t,t.i[e][2])))),I=true,A>P&&A>r?P+r<n1&&(I=false,L=Lt-.5*(n.Pl(t,t.i[e][0])+n.Pl(t,t.i[e][1])),b=ot-.5*(n.Ql(t,t.i[e][0])+n.Ql(t,t.i[e][1]))):P>A&&P>r?A+r<n1&&(I=false,L=Lt-.5*(n.Pl(t,t.i[e][0])+n.Pl(t,t.i[e][2])),b=ot-.5*(n.Ql(t,t.i[e][0])+n.Ql(t,t.i[e][2]))):A+P<n1&&(I=false,L=Lt-.5*(n.Pl(t,t.i[e][1])+n.Pl(t,t.i[e][2])),b=ot-.5*(n.Ql(t,t.i[e][1])+n.Ql(t,t.i[e][2]))),I){for(Qt=t.i[e][0],Q=L0,d=0;d<3;d++)i=t.i[e][d],i!=ut&&(o=j.Math.pow(t.J[e].a-t.J[i].a,2)+j.Math.pow(t.J[e].b-t.J[i].b,2),o<Q&&(Qt=i,Q=o,T.Ujb()));v=n.Sk(t,(t.J[ut].a+t.J[Qt].a)/2,(t.J[ut].b+t.J[Qt].b)/2,0);}else v=n.Sk(t,Lt+L,ot+b,0);}else v=n.Sk(t,Lt+L,ot+b,0);n.Bn(t,v,1),n.Vk(t,e,v,1);}break;case 2:if(K=t.j[e],K==1)L=Lt-n.Pl(t,t.i[e][0]),b=ot-n.Ql(t,t.i[e][0]),v=n.Sk(t,Lt+(Yt.Dz*L-Yt.Iz*b)*.7,ot+(Yt.Iz*L+Yt.Dz*b)*.7,0),n.Bn(t,v,1),n.Vk(t,e,v,1),v=n.Sk(t,Lt+(Yt.Az*L-Yt.Fz*b)*.7,ot+(Yt.Fz*L+Yt.Az*b)*.7,0),n.Bn(t,v,1),n.Vk(t,e,v,1);else if(K==2)s=Lt-n.Pl(t,t.i[e][0]),c=ot-n.Ql(t,t.i[e][0]),h=Lt-n.Pl(t,t.i[e][1]),g=ot-n.Ql(t,t.i[e][1]),S=j.Math.sqrt(s*s+c*c)*.7,V=j.Math.sqrt(h*h+g*g)*.7,L=s+h,b=c+g,M=j.Math.sqrt(L*L+b*b),D=(S+V)/2,L=L/M*D,b=b/M*D,Dt=n.Hp(t,e),v=n.Sk(t,Lt+Yt.zz*L-Yt.Ez*b,ot+Yt.Ez*L+Yt.zz*b,0),n.Bn(t,v,1),Dt>-1?n.Vk(t,e,v,1):n.Vk(t,e,v,257),v=n.Sk(t,Lt+Yt.Bz*L-Yt.Gz*b,ot+Yt.Gz*L+Yt.Bz*b,0),n.Bn(t,v,1),n.Vk(t,e,v,1);else for(C=0;C<O;C++)v=n.Sk(t,Lt,ot,0),n.Bn(t,v,1),n.Vk(t,e,v,1);break;case 3:t.i[e][0]>0?(L=(Lt-n.Pl(t,t.i[e][0]))*.7,b=(ot-n.Ql(t,t.i[e][0]))*.7,v=n.Sk(t,Lt+L,ot+b,0),n.Bn(t,v,1),n.Vk(t,e,v,1),v=n.Sk(t,Lt-b,ot+L,0),n.Bn(t,v,1),n.Vk(t,e,v,1),v=n.Sk(t,Lt+b,ot-L,0),n.Bn(t,v,1),n.Vk(t,e,v,1)):(L=n.Uo(t,true),b=n.Uo(t,true),v=n.Sk(t,Lt+L,ot+b,0),n.Bn(t,v,1),n.Vk(t,e,v,1),v=n.Sk(t,Lt-b*Yt.Dz,ot+L*Yt.Iz,0),n.Bn(t,v,1),n.Vk(t,e,v,1),v=n.Sk(t,Lt-b*Yt.Dz,ot-L*Yt.Iz,0),n.Bn(t,v,1),n.Vk(t,e,v,1));break;default:{for(C=0;C<O;C++)v=n.Sk(t,Lt,ot,0),n.Bn(t,v,1),n.Vk(t,e,v,1);break;}}},Yt.Mz=function(t){Yt.Jz();var e,A;for(n.aw(t,1),A=0,e=0;e<t.q;e++)t.C[e]==1?++A:A+=t.e[e]-t.j[e]+n.np(t,e);return A;},Yt.zz=0,Yt.Az=0,Yt.Bz=0,Yt.Cz=0,Yt.Dz=0,Yt.Ez=0,Yt.Fz=0,Yt.Gz=0,Yt.Hz=0,Yt.Iz=0,Z.Oz=function(t,e,A){var P,r,i;for(i=e.length,P=new Z.tB(t.f,i,false),P.a[0]=0,P.b[0]=0,r=0;r<i;r++)P.k[r]=128-i,P.e[r]=e[r];i<8?Z.Xz(P):Z.Vz(t,P,e,A),Z.EA(t.d,P);},Z.Pz=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h;for(D=z(l.d7,H,6,r,15,1),i=0;i<t.f.q;i++){for(o=0;o<n.Ko(t.f,i);o++){for(h=A[n.Yo(t.f,i,o)],Q=0;Q<o&&!(h<D[Q]);Q++);for(L=o;L>Q;L--)D[L]=D[L-1];D[Q]=h;}for(s=n.Ko(t.f,i),n.dh(e[i],i),n._g(e[i],P,l.Oeb(A[i])),n._g(e[i],(r-s)*(P+1),0),u=0;u<s;u++)n._g(e[i],P+1,l.Oeb(D[u]));}},Z.Qz=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(P=n.Og(t.f.f),o=2,A=0;A<t.f.f;A++)o=j.Math.max(o,n.Ko(t.f,A));for(i=(62+2*P+o*(P+1))/63|0,r=z(l.p7,Mn,101,t.f.q,0,1),e=0;e<t.f.q;e++)r[e]=new n.eh(i),n.dh(r[e],e);for(s=z(l.d7,H,6,t.f.q,15,1),D=0;D<t.f.g;D++)u=n.fm(t.f,D),(u==1||u==2)&&(n.ah(r[n.Yl(t.f,0,D)],l.Oeb(u)),n.ah(r[n.Yl(t.f,1,D)],l.Oeb(u)));Q=Z.Rz(r,s);do L=Q,Z.Pz(t,r,s,P,o),Q=Z.Rz(r,s);while(L!=Q);return s;},Z.Rz=function(t,e){var A,P;for(P=0,U.Wqb(t,0,t.length,null),A=0;A<t.length;A++)(A==0||n.bh(t[A],t[A-1])!=0)&&++P,e[t[A].a]=P;return P;},Z.Sz=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h;for(o=new U.wqb(t.d);o.a<o.c.a.length;)for(u=U.vqb(o),Q=0;Q<u.f.length;Q++)if(P=u.f[Q],n.em(t.f,P)==2&&(!n.iq(t.f,P)&&(n.fm(t.f,P)==3||n.fm(t.f,P)==0)&&n.Jn(t.f,P),!n.fq(t.f,P)&&n.Zo(t.f,n.Yl(t.f,0,P))>1&&n.Zo(t.f,n.Yl(t.f,1,P))>1&&(n.fm(t.f,P)==1||n.fm(t.f,P)==2))){for(h=z(l.d7,H,6,2,15,1),r=z(l.d7,H,6,2,15,1),L=0;L<2;L++)for(h[L]=t.f.M,r[L]=n.Yl(t.f,L,P),s=0;s<n.Ko(t.f,r[L]);s++)i=n.Yo(t.f,r[L],s),i!=n.Yl(t.f,1-L,P)&&h[L]>i&&(h[L]=i);D=Z.ZA(u.a[u.g[r[0]]],u.b[u.g[r[0]]],u.a[u.g[r[1]]],u.b[u.g[r[1]]]),e=Z.ZA(u.a[u.g[h[0]]],u.b[u.g[h[0]]],u.a[u.g[r[0]]],u.b[u.g[r[0]]]),A=Z.ZA(u.a[u.g[r[1]]],u.b[u.g[r[1]]],u.a[u.g[h[1]]],u.b[u.g[h[1]]]),Z.Zz(D,e)<0^Z.Zz(D,A)<0^n.fm(t.f,P)==2&&Z.hB(u,P);}},Z.Tz=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;for(h=new U.Lj(),m=new U.Lj(),P=new U.Lj(),i=new U.wqb(t.d);i.a<i.c.a.length;){for(r=U.vqb(i),D=0,w.pzb(P.a,0),u=0;u<r.e.length;u++)e=r.e[u],A=t.i[e],A!=0&&(U.rj(P,new Z._A(r,u,A)),D+=A);if(D!=0)for(U.Ij(P,new Z.QA()),Q=new U.wqb(P);Q.a<Q.c.a.length&&(o=U.vqb(Q),!(D*o.b>0&&(A=j.Math.abs(D)>=j.Math.abs(o.b)?o.b:D,D-=A,U.rj(A<0?h:m,new Z._A(r,o.a,A)),D==0))););}if(h.a.length==0||m.a.length==0)return null;for(U.Ij(m,new Z.JA()),U.Ij(h,new Z.NA()),g=new U.wqb(m);g.a<g.c.a.length;)for(b=U.vqb(g),s=new U.wqb(h);s.a<s.c.a.length;)if(L=U.vqb(s),b.b==-L.b)return t.i[Z.lB(b.c,b.a)]-=b.b,t.i[Z.lB(L.c,L.a)]-=L.b,new Z.WA(b.c,L.c,b.a,L.a);for(d=new U.wqb(m);d.a<d.c.a.length;)for(b=U.vqb(d),s=new U.wqb(h);s.a<s.c.a.length;)if(L=U.vqb(s),b.b>-L.b)return t.i[Z.lB(b.c,b.a)]+=L.b,t.i[Z.lB(L.c,L.a)]-=L.b,new Z.WA(b.c,L.c,b.a,L.a);for(c=new U.wqb(m);c.a<c.c.a.length;)for(b=U.vqb(c),s=new U.wqb(h);s.a<s.c.a.length;)if(L=U.vqb(s),b.b<-L.b)return t.i[Z.lB(b.c,b.a)]-=b.b,t.i[Z.lB(L.c,L.a)]+=b.b,new Z.WA(b.c,L.c,b.a,L.a);return null;},Z.Uz=function(t){return t.d.a.length<2?null:new Z.VA(U.wj(t.d,0),U.wj(t.d,1));},Z.Vz=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N;if(L=F(k(l.b7,2),l1,15,0,[F(k(l.b7,1),jt,6,15,[20]),null,null,F(k(l.b7,1),jt,6,15,[0,10]),null,null,F(k(l.b7,1),jt,6,15,[-4,12]),F(k(l.b7,1),jt,6,15,[0,0,-7.5]),null,null,null,null,F(k(l.b7,1),jt,6,15,[8.571428571428571,-8.571428571428571]),null,null,null,F(k(l.b7,1),jt,6,15,[-2.4])]),s=F(k(l.d7,2),S1,7,0,[F(k(l.d7,1),H,6,15,[146]),F(k(l.d7,1),H,6,15,[627]),null,F(k(l.d7,1),H,6,15,[2457,1170]),null,F(k(l.d7,1),H,6,15,[2451,8643,2519]),F(k(l.d7,1),H,6,15,[9362,14798]),F(k(l.d7,1),H,6,15,[34377,-2147448999,26214]),null,F(k(l.d7,1),H,6,15,[37449,137313,95703,34371,37815,54891,132867,-2147309741,54857,55129,-2147449005,-2147449065]),null,F(k(l.d7,1),H,6,15,[530697,531819,899169,137289,694617,-2146951863,-2146952797,-2146939175,-2146929547,-2146929564,-2146625111,-2146931799,-2146940503,-2146931935]),F(k(l.d7,1),H,6,15,[1007293,610915]),F(k(l.d7,1),H,6,15,[542985,137283,2122017,530691,2206773,-2144711351,219209,2840841,137555,-2146871031,-2147264167,613705,-2145360543,-2146625271,694611,2454837,-2145356703,-2147345133,-2146928951,-2146931805,-2144641719,-2146951869,-2146625237,-2146624183,2841963,1074905,-2146625117,2799955,-2144723645,138583,859225,-2145264843,-2145216253,-2146624149,-2144700727,-2146928917,-2143905527,-2144045771,-2146789097,2288547,544407,2104323,-2146911977,-2144479405,3633737,-2146870089,-2146952169]),null,F(k(l.d7,1),H,6,15,[8487297,2172633,2116611,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8829813]),F(k(l.d7,1),H,6,15,[14071213])]),C=1<<e.e.length,D=0,o=0,e.e.length>7)for(b=0;b<e.e.length;b++)u=Z.bA(t,A,P,b),u==1?D+=C:u==2&&(o+=C),D>>>=1,o>>>=1;if(S=e.e.length-9,e.e.length>=9&&e.e.length<=25&&s[S]!=null)for(N=0;N<s[S].length;N++)for(m=(P1&s[S][N])==0,Q=St&s[S][N],g=false;!g;g=!g){if(g){if(m)break;for(v=0,r=1;r!=C;r<<=1)v<<=1,(Q&r)!=0&&(v|=1);Q=v;}for(V=0;V<e.e.length;V++){if((Q&D)==0&&(~Q&o)==0){for(i=0,h=Ai*(L[S]==null?0:L[S][N]),M=0,O=Q,d=true,c=0;c<e.e.length;c++)d&&++M,(O&1)==0&&(d=!d),O>>>=1;for(K=M>(e.e.length/2|0),b=1;b<e.e.length;b++)e.a[b]=e.a[b-1]+j.Math.sin(i),e.b[b]=e.b[b-1]+j.Math.cos(i),(Q&1)==0&&(K=!K),i+=h+(K?b0:cA),Q>>>=1;return;}(Q&1)!=0&&(Q|=C),Q>>>=1;}}Z.Yz(e,D,o);},Z.Wz=function(t){var e,A,P,r,i,D,u,o,Q,L;for(e=null,Q=null,D=0;D<t.f.g;D++)if(n.hm(t.f,D)==16){for(A=n.Yl(t.f,0,D),r=-1,u=0;u<t.d.a.length&&(r=Z.mB(U.wj(t.d,u),A),r==-1);u++);for(P=n.Yl(t.f,1,D),i=-1,o=0;o<t.d.a.length&&(i=Z.mB(U.wj(t.d,o),P),i==-1);o++);u!=o&&(u>o&&(L=u,u=o,o=L,L=r,r=i,i=L),Q==null&&(Q=z(l.w8,c0,450,t.d.a.length,0,2)),Q[o]==null&&(Q[o]=z(l.w8,{450:1,4:1,5:1,8:1},106,o,0,1)),Q[o][u]?Z.TA(Q[o][u],r,i):(a1(Q[o],u,new Z.WA(U.wj(t.d,u),U.wj(t.d,o),r,i)),!e&&(e=new U.Lj()),U.rj(e,Q[o][u])));}return e;},Z.Xz=function(t){var e,A;for(e=n1-n1*(t.e.length-2)/t.e.length,A=1;A<t.e.length;A++)t.a[A]=t.a[A-1]+j.Math.sin(e*(A-1)),t.b[A]=t.b[A-1]+j.Math.cos(e*(A-1));},Z.Yz=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;if(e==0||(e&A)!=0){Z.Xz(t);return;}for(d=-1,m=0,i=1<<t.e.length-2,r=1<<t.e.length-1,u=1,D=2,Q=0;Q<t.e.length;Q++)(A&(r|u))==0&&(e&(r|u))!=0&&(e&i)==0&&(c=0,(A&i)!=0&&(c+=4),(e&r)!=0&&(c+=2),(e&u)!=0&&(c+=1),m<c&&(m=c,d=Q)),i=r,r=u,u=D,D=1<<(Q+2<t.e.length?Q+2:Q+2-t.e.length);if(d==-1){Z.Xz(t);return;}for(h=0,h|=1<<d,b=2;b<t.e.length-1;){if(s=d+b<t.e.length?d+b:d+b-t.e.length,r=1<<(s==0?t.e.length-1:s-1),(A&r)!=0){++b;continue;}if(u=1<<s,(e&r)!=0){if((A&u)!=0){Z.Xz(t);return;}h|=u,b+=2;continue;}if(D=1<<(s+1<t.e.length?s+1:s+1-t.e.length),(e&u)!=0&&(A&D)!=0){h|=u,b+=3;continue;}++b;}if(h==0){Z.Xz(t);return;}for(P=n1-n1*(t.e.length-2)/t.e.length,L=1;L<t.e.length;L++)t.a[L]=t.a[L-1]+j.Math.sin(P*(L-1)),t.b[L]=t.b[L-1]+j.Math.cos(P*(L-1));for(u=1,g=2*j.Math.sin(P/2),o=0;o<t.e.length;o++)(h&u)!=0&&(t.a[o]+=g*j.Math.cos(P*(o-.5)),t.b[o]-=g*j.Math.sin(P*(o-.5))),u<<=1;},Z.Zz=function(t,e){var A;for(A=t-e;A<lr;)A+=X1;for(;A>n1;)A-=X1;return A;},Z.$z=function(t,e,A){var P,r;for(P=0,r=0;r<n.Ko(t.f,A);r++)Z.oB(e,n.Yo(t.f,A,r))&&++P;return P;},Z._z=function(t,e,A,P){var r,i,D,u,o;return u=Z.mB(e,P),o=Z.mB(A,P),Z.sB(A,e.a[u]-A.a[o],e.b[u]-A.b[o]),r=Z.yA(t,e,P),i=Z.yA(t,A,P),D=0,Z.$z(t,e,P)==1&&Z.$z(t,A,P)==1&&(D=b0),Z.rB(A,A.a[o],A.b[o],r-i+D+n1),Z.eA(t,e,A,1);},Z.aA=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt;for(v=z(l.d7,H,6,r,15,1),M=z(l.d7,H,6,r,15,1),g=0;g<r;g++)v[g]=Z.mB(e,P[g]),M[g]=Z.mB(A,P[g]);for(ut=0,Dt=0,Qt=0,Lt=0,d=0;d<r;d++)ut+=e.a[v[d]],Dt+=e.b[v[d]],Qt+=A.a[M[d]],Lt+=A.b[M[d]];for(ut/=r,Dt/=r,Qt/=r,Lt/=r,Z.sB(A,ut-Qt,Dt-Lt),Q=z(l.x8,C1,41,r,0,1),s=z(l.x8,C1,41,r,0,1),i=z(l.x8,C1,41,r,0,1),D=z(l.x8,C1,41,r,0,1),m=0;m<r;m++)Q[m]=new Z.YA(ut,Dt,e.a[v[m]],e.b[v[m]]),s[m]=new Z.YA(ut,Dt,A.a[M[m]],A.b[M[m]]),i[m]=new Z.XA(Q[m].a-s[m].a,Q[m].b*s[m].b),D[m]=new Z.XA(Q[m].a+s[m].a,Q[m].b*s[m].b);for(V=Z.CA(i,r),O=Z.CA(D,r),rt=0,yt=0,C=0;C<r;C++)for(S=0;S<n.Ko(t.f,P[C]);S++)u=n.Yo(t.f,P[C],S),Z.oB(e,u)&&!Z.oB(A,u)&&++rt,!Z.oB(e,u)&&Z.oB(A,u)&&++yt;for(L=z(l.x8,C1,41,rt,0,1),h=z(l.x8,C1,41,yt,0,1),b=z(l.x8,C1,41,yt,0,1),rt=0,yt=0,c=0;c<r;c++)for(S=0;S<n.Ko(t.f,P[c]);S++)u=n.Yo(t.f,P[c],S),Z.oB(e,u)&&!Z.oB(A,u)&&(o=Z.mB(e,u),L[rt]=new Z.YA(e.a[v[c]],e.b[v[c]],e.a[o],e.b[o]),++rt),!Z.oB(e,u)&&Z.oB(A,u)&&(o=Z.mB(A,u),ot=new Z.YA(A.a[M[c]],A.b[M[c]],A.a[o],A.b[o]),h[yt]=new Z.XA(V.a+ot.a,ot.b),b[yt]=new Z.XA(O.a-ot.a,ot.b),++yt);return K=Z.CA(L,rt),N=Z.CA(h,yt),I=Z.CA(b,yt),j.Math.abs(Z.Zz(K.a,N.a))>j.Math.abs(Z.Zz(K.a,I.a))?Z.rB(A,ut,Dt,V.a):(Z.gB(A,ut,Dt),Z.rB(A,ut,Dt,O.a)),Z.eA(t,e,A,r);},Z.bA=function(t,e,A,P){var r,i,D,u,o,Q;return i=P==e.length-1?0:P+1,u=P==0?e.length-1:P-1,D=i==e.length-1?0:i+1,n.em(t.f,A[P])==2&&(r=n.fm(t.f,A[P]),r==1||r==2)?(Z.jA(t,e[u],e[P],e[i])^Z.jA(t,e[D],e[i],e[P])&&(r=r==1?2:1),r):n.iq(t.f,A[P])?(o=n.kt(n.Dp(t.f),A[u],A[P]),Q=n.kt(n.Dp(t.f),A[i],A[P]),o!=-1||Q!=-1?o==Q?2:1:2):0;},Z.cA=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b;for(r=z(l.d7,H,6,t.f.q,15,1),i=z(l.d7,H,6,t.f.q,15,1),D=z(l.d7,H,6,t.f.q,15,1),u=z(l.d7,H,6,t.f.q,15,1),r[0]=e,D[e]=1,u[0]=-1,P=0,o=0;P<=o;){if(P==0||!t.b[r[P]])for(Q=0;Q<n.Ko(t.f,r[P]);Q++)A=n.Yo(t.f,r[P],Q),h=n.$o(t.f,r[P],Q),D[A]==0&&!t.c[h]&&(r[++o]=A,i[o]=h,D[A]=D[r[P]]+1,u[o]=P);if(P==o){for(b=new Z.$A(D[r[P]]),L=P,s=0;s<b.a.length;s++)b.a[s]=r[L],b.b[s]=i[L],L=u[L];return b;}++P;}return null;},Z.dA=function(t){var e,A,P,r;for(r=0,P=null,A=new U.wqb(t);A.a<A.c.a.length;)e=U.vqb(A),r<e.b[0].e.length*e.b[1].e.length&&(r=e.b[0].e.length*e.b[1].e.length,P=e);return P;},Z.eA=function(t,e,A,P){var r,i,D,u,o;for(i=new Z.tB(t.f,e.e.length+A.e.length-P,e.i|A.i),r=0,u=0;u<e.e.length;u++)i.e[r]=e.e[u],i.k[r]=e.k[u],i.a[r]=e.a[u],i.b[r++]=e.b[u];for(D=0;D<A.e.length;D++)o=Z.mB(e,A.e[D]),o==-1?(i.e[r]=A.e[D],i.k[r]=A.k[D],i.a[r]=A.a[D],i.b[r++]=A.b[D]):i.k[o]<A.k[D]&&(i.k[o]=A.k[D],i.a[o]=A.a[D],i.b[o]=A.b[D]);return i;},Z.fA=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;for(u=z(l.d7,H,6,t.f.q,15,1),o=z(l.d7,H,6,t.f.q,15,1),Q=z(l.d7,H,6,t.f.q,15,1),L=z(l.d7,H,6,t.f.q,15,1),u[0]=A,Q[A]=1,L[0]=-1,D=0,s=0;D<=s;){for(h=0;h<n.Ko(t.f,u[D]);h++){if(r=n.Yo(t.f,u[D],h),c=n.$o(t.f,u[D],h),r==e){for(i=Q[u[D]],P=z(l.d7,H,6,i,15,1),P[0]=c,b=1;b<i;b++)P[b]=o[D],D=L[D];return P;}Q[r]==0&&(u[++s]=r,o[s]=c,Q[r]=Q[u[D]]+1,L[s]=D);}if(D==s)return null;++D;}return null;},Z.gA=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c;for(A=n.Yl(t.f,0,e),P=n.Yl(t.f,1,e),D=z(l.d7,H,6,t.f.q,15,1),u=z(l.d7,H,6,t.f.q,15,1),o=z(l.d7,H,6,t.f.q,15,1),Q=z(l.d7,H,6,t.f.q,15,1),D[0]=A,D[1]=P,u[1]=e,o[A]=1,o[P]=2,Q[0]=-1,Q[1]=0,i=1,L=1;i<=L;){for(s=0;s<n.Zo(t.f,D[i]);s++){if(r=n.Yo(t.f,D[i],s),i>1&&r==A){for(c=new Z.$A(o[D[i]]),u[0]=n.$o(t.f,D[i],s),h=i,b=0;b<c.a.length;b++)c.a[b]=D[h],c.b[b]=u[h],h=Q[h];return c;}o[r]==0&&n.eq(t.f,r)&&(D[++L]=r,u[L]=n.$o(t.f,D[i],s),o[r]=o[D[i]]+1,Q[L]=i);}++i;}return null;},Z.hA=function(t,e,A,P){var r,i,D,u,o,Q;for(D=z(l.d7,H,6,t.f.q,15,1),u=z(l.d7,H,6,t.f.q,15,1),D[0]=A,D[1]=e,u[A]=1,u[e]=2,i=1,o=1;i<=o;){for(Q=0;Q<n.Zo(t.f,D[i]);Q++){if(r=n.Yo(t.f,D[i],Q),r==P)return 1+u[D[i]];u[r]==0&&n.eq(t.f,r)&&(D[++o]=r,u[r]=u[D[i]]+1);}++i;}return 0;},Z.iA=function(t,e){var A,P,r,i,D,u,o;for(u=(e.T&15)!=0,o=e.T&248,!t.g&&(t.g=new U.yvb()),(t.e&2)!=0&&n.pq(e,false),t.f=e,n.aw(t.f,7),t.d=new Z.GA(),t.b=z(l.zeb,ht,6,t.f.q,16,1),t.c=z(l.zeb,ht,6,t.f.r,16,1),t.i=z(l.d7,H,6,t.f.q,15,1),A=0;A<t.f.q;A++)t.i[A]=n.Al(t.f,A);for((t.e&12)!=0&&Z.tA(t),(t.e&1)==0&&Z.Nz&&Z.vA(t,Z.Nz),Z.sA(t),Z.oA(t),Z.qA(t),Z.oA(t),i=new U.wqb(t.d);i.a<i.c.a.length;)P=U.vqb(i),Z.pB(P);for(Z.Sz(t),Z.wA(t),Z.uA(t),Z.nA(t),Z.lA(t),Z.pA(t),r=new U.wqb(t.d);r.a<r.c.a.length;)for(P=U.vqb(r),D=0;D<P.e.length;D++)n.yn(t.f,P.e[D],P.a[D]),n.zn(t.f,P.e[D],P.b[D]),n.An(t.f,P.e[D],0);u&&(n.uq(t.f,o),n.xq(t.f)),t.a&&n.Ym(t.f);},Z.jA=function(t,e,A,P){var r,i;for(i=0;i<n.Zo(t.f,A);i++)if(r=n.Yo(t.f,A,i),r!=P&&r<e)return  false;return  true;},Z.kA=function(t,e,A){var P;Z.UA(e,A,(t.e&12)!=0),P=Z.eA(t,e.b[0],e.b[1],0),Z.zA(t,e.b[0],e.b[1],P);},Z.lA=function(t){var e;for(e=Z.Tz(t);e;)Z.kA(t,e,UA),e=Z.Tz(t);},Z.mA=function(t,e,A,P){var r,i,D,u,o;for(r=z(l.d7,H,6,P,15,1),i=0,D=0;D<e.e.length;D++)for(u=0;u<A.e.length;u++)e.e[D]==A.e[u]&&(r[i++]=e.e[D]);o=P==1?Z._z(t,e,A,r[0]):Z.aA(t,e,A,r,P),Z.zA(t,e,A,o);},Z.nA=function(t){var e,A;for(A=Z.Wz(t);A;)e=Z.dA(A),Z.kA(t,e,hA),A=Z.Wz(t);},Z.oA=function(t){for(var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;;){for(C=0,b=0,d=null,m=null,D=1;D<t.d.a.length;D++)for(P=U.wj(t.d,D),u=0;u<D;u++){for(r=U.wj(t.d,u),e=0,A=0,c=0,g=0,L=0;L<P.e.length;L++)for(h=0;h<r.e.length;h++)P.e[L]==r.e[h]&&(++A,e=P.e[L],c<P.k[L]&&(c=P.k[L]),g<r.k[h]&&(g=r.k[h]));if(A>0&&(i=A==1&&Z.$z(t,P,e)==1&&Z.$z(t,r,e)==1?0:1,c>g?o=(i<<24)+(c<<16)+(g<<8)+A:o=(i<<24)+(g<<16)+(c<<8)+A,C<o)){for(C=o,b=A,c=0,g=0,s=0;s<P.e.length;s++)c<P.k[s]&&(c=P.k[s]);for(Q=0;Q<r.e.length;Q++)g<r.k[Q]&&(g=r.k[Q]);c>g?(d=P,m=r):(d=r,m=P);}}if(C==0)break;b==d.e.length?U.Ej(t.d,d):b==m.e.length?U.Ej(t.d,m):Z.mA(t,d,m,b);}},Z.pA=function(t){var e;for(e=Z.Uz(t);e;)Z.kA(t,e,C9),e=Z.Uz(t);},Z.qA=function(t){for(var e,A,P,r,i,D,u;;){for(i=null,e=0;e<t.f.q;e++){for(u=0,r=0;r<n.Ko(t.f,e);r++)t.c[n.$o(t.f,e,r)]||++u;u==1&&(D=Z.cA(t,e),(!i||D.a.length>i.a.length)&&(i=D));}if(!i)break;for(A=new Z.tB(t.f,i.a.length,false),P=0;P<i.a.length;P++)t.b[i.a[P]]=true,P<i.a.length-1&&(t.c[i.b[P]]=true),A.e[P]=i.a[P],A.a[P]=j.Math.cos(Hn)*P,A.b[P]=(P&1)==1?0:.5,A.k[P]=128+i.a.length;Z.EA(t.d,A);}},Z.rA=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h;for(i=0;i<t.f.r;i++)if(P=n.Yl(t.f,0,i),r=n.Yl(t.f,1,i),!(n.fq(t.f,i)||n.em(t.f,i)!=1||n.Ko(t.f,P)==1||n.Ko(t.f,r)==1)&&!((t.e&4)!=0&&n.Jm(t.f,P)&&n.Jm(t.f,r))){for(s=false,Q=0;Q<2;Q++)if(D=n.Yl(t.f,Q,i),n.Ko(t.f,D)>2){for(h=true,o=-1,L=0;L<n.Ko(t.f,D);L++)u=n.Yo(t.f,D,L),u!=n.Yl(t.f,1-Q,i)&&(o==-1?o=A[u]:o!=A[u]&&(h=false));if(h){s=true;break;}}s||((t.e&8)!=0&&n.Jm(t.f,P)&&n.Jm(t.f,r)?e[i]=1:e[i]=2);}},Z.sA=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt;for(P=0;P<t.f.f;P++)if(n.Ko(t.f,P)>4){for(h=new Z.tB(t.f,1+n.Ko(t.f,P),false),h.a[n.Ko(t.f,P)]=0,h.b[n.Ko(t.f,P)]=0,h.k[n.Ko(t.f,P)]=32,h.e[n.Ko(t.f,P)]=P,t.b[P]=true,c=0;c<n.Ko(t.f,P);c++)Q=n.Yo(t.f,P,c),h.a[c]=j.Math.sin(b0*c-En),h.b[c]=j.Math.cos(b0*c-En),h.k[c]=32,h.e[c]=Q,t.b[Q]=true,t.c[n.$o(t.f,P,c)]=true;Z.EA(t.d,h);}for(Lt=n.Dp(t.f),Dt=0;Dt<Lt.j.a.length;Dt++){if(ot=U.wj(Lt.k,Dt).length,ut=U.wj(Lt.j,Dt),rt=false,(t.e&12)!=0){for(rt=true,c=0;c<ot;c++)if(!n.Jm(t.f,ut[c])){rt=false;break;}}if(!rt){for(m=false,g=0;g<ot;g++)if(n.To(t.f,ut[g])==ot){m=true;break;}if(m)for(Qt=U.wj(Lt.k,Dt),Z.Oz(t,ut,Qt),c=0;c<ot;c++)t.b[ut[c]]=true,t.c[Qt[c]]=true;}}for(u=0;u<t.f.g;u++)if(n.fq(t.f,u)&&!t.c[u])for(bt=Z.gA(t,u),ut=bt.a,Qt=bt.b,Z.Oz(t,ut,Qt),c=0;c<bt.a.length;c++)t.b[ut[c]]=true,t.c[Qt[c]]=true;for(o=0;o<t.f.r;o++)if(!t.c[o]&&n.em(t.f,o)==3&&(r=n.Yl(t.f,0,o),i=n.Yl(t.f,1,o),V=n.Ko(t.f,r)+n.Ko(t.f,i),V>2)){for(h=new Z.tB(t.f,V,false),L=0,g=0;g<n.Ko(t.f,r);g++)Q=n.Yo(t.f,r,g),Q!=i&&(h.e[L++]=Q,t.b[Q]=true,t.c[n.$o(t.f,r,g)]=true);for(h.e[L++]=r,h.e[L++]=i,d=0;d<n.Ko(t.f,i);d++)Q=n.Yo(t.f,i,d),Q!=r&&(h.e[L++]=Q,t.b[Q]=true,t.c[n.$o(t.f,i,d)]=true);for(c=0;c<V;c++)h.a[c]=c,h.b[c]=0,h.k[c]=1;t.b[r]=true,t.b[i]=true,t.c[o]=true,Z.EA(t.d,h);}for(D=0;D<t.f.r;D++)if(!t.c[D]&&n.em(t.f,D)==2){for(e=z(l.d7,H,6,t.f.q,15,1),c=0;c<2;c++)if(e[0]=n.Yl(t.f,c,D),e[1]=n.Yl(t.f,1-c,D),n.Po(t.f,e[0])==1&&n.Po(t.f,e[1])==2&&n.Ko(t.f,e[1])==2){t.b[e[0]]=true,t.b[e[1]]=true,t.c[D]=true,S=1;do{if(O=n.Yo(t.f,e[S],0)==e[S-1]?1:0,e[S+1]=n.Yo(t.f,e[S],O),n.Po(t.f,e[S+1])==2&&n.Ko(t.f,e[S+1])>2)break;t.b[e[S+1]]=true,t.c[n.$o(t.f,e[S],O)]=true,++S;}while(n.Po(t.f,e[S])==2&&n.Ko(t.f,e[S])==2);for(V=n.Ko(t.f,e[0])+n.Ko(t.f,e[S])+S-1,h=new Z.tB(t.f,V,false),v=0;v<=S;v++)h.a[v]=v,h.b[v]=0,h.k[v]=64,h.e[v]=e[v];for(s=S+1,b=false,M=0;M<n.Ko(t.f,e[0]);M++)Q=n.Yo(t.f,e[0],M),Q!=e[1]&&(h.a[s]=-0.5,h.b[s]=b?j.Math.sin(b0):-j.Math.sin(b0),h.k[s]=64,h.e[s]=Q,++s,b=true);for(b=false,C=0;C<n.Ko(t.f,e[S]);C++)Q=n.Yo(t.f,e[S],C),Q!=e[S-1]&&(h.a[s]=S+.5,h.b[s]=b?-j.Math.sin(b0):j.Math.sin(b0),h.k[s]=64,h.e[s]=Q,++s,b=true);Z.EA(t.d,h);}}for(A=0;A<t.f.q;A++)if(n.Ko(t.f,A)==4){for(K=z(l.d7,H,6,4,15,1),N=z(l.d7,H,6,4,15,1),I=0,g=0;g<4;g++)K[I]=n.Yo(t.f,A,g),N[I]=n.$o(t.f,A,g),n.Ko(t.f,K[I])==1&&!t.c[N[I]]&&++I;if(I==2){for(h=new Z.tB(t.f,3,false),c=0;c<2;c++)t.b[K[c]]=true,t.c[N[c]]=true,h.e[c]=K[c],h.k[c]=32;h.a[0]=-0.5,h.b[0]=.866,h.a[1]=.5,h.b[1]=.866,h.a[2]=0,h.b[2]=0,h.k[2]=32,h.e[2]=A,Z.EA(t.d,h);}if(I==3){for(d=0;d<2;d++)n.em(t.f,N[d])==1&&(yt=K[d],K[d]=K[2],K[2]=yt,yt=N[d],N[d]=N[2],N[2]=yt);for(h=new Z.tB(t.f,4,false),c=0;c<3;c++)t.b[K[c]]=true,t.c[N[c]]=true,h.e[c]=K[c],h.k[c]=32;h.a[0]=-1,h.b[0]=0,h.a[1]=1,h.b[1]=0,h.a[2]=0,h.b[2]=1,h.a[3]=0,h.b[3]=0,h.k[3]=32,h.e[3]=A,Z.EA(t.d,h);}}},Z.tA=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;for(u=0,A=0;A<t.f.q;A++)n.Jm(t.f,A)&&++u;if(!(u<2)){for(s=0,Q=0,L=0;L<t.f.r;L++)P=n.Yl(t.f,0,L),r=n.Yl(t.f,1,L),n.Jm(t.f,P)&&n.Jm(t.f,r)&&(t.c[L]=true,t.b[P]=true,t.b[r]=true,Q+=n.cm(t.f,L),++s);for(s!=0&&Q!=0?Q/=s:Q=n.Tl(t.f),i=0;i<t.f.q;i++)n.Jm(t.f,i)&&!t.b[i]&&--u;if(!(u<2)){for(v=z(l.d7,H,6,t.f.q,15,1),h=n.gp(t.f,v,true,true),C=z(l.d7,H,6,h,15,1),D=0;D<t.f.q;D++)v[D]!=-1&&++C[v[D]];for(m=z(l.B8,C1,47,h,0,1),c=0;c<h;c++)m[c]=new Z.tB(t.f,C[c],true);for(o=z(l.d7,H,6,h,15,1),e=0;e<t.f.q;e++)g=v[e],g!=-1&&(m[g].k[o[g]]=i0,m[g].e[o[g]]=e,m[g].a[o[g]]=n.Pl(t.f,e)/Q,m[g].b[o[g]]=n.Ql(t.f,e)/Q,++o[g]);for(M=-1,S=0,d=0;d<h;d++)S<C[d]&&(S=C[d],M=d);for(Z.EA(t.d,m[M]),b=0;b<h;b++)b!=M&&Z.EA(t.d,m[b]);}}},Z.uA=function(t){var e,A;for(e=0;e<t.f.q;e++)!t.b[e]&&n.Ko(t.f,e)==0&&(A=new Z.tB(t.f,1,false),t.b[e]=true,A.e[0]=e,A.a[0]=0,A.b[0]=0,A.k[0]=0,Z.EA(t.d,A));},Z.vA=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;for(c=null,c=new n.hu(),n.bu(c,t.f),D=z(l.zeb,ht,6,t.f.f,16,1),d=e.Sb();d.cd();)if(g=d.dd(),b=null,C=g.b,n.$t(c,C),n.Ot(c,4,8)!=0&&(b=c.F),b)for(h=new U.wqb(b);h.a<h.c.a.length;){for(s=U.vqb(h),m=0,P=s,r=0,i=P.length;r<i;++r)A=P[r],D[A]&&++m;if(m<=1){for(o=g.c,t.a?o=false:t.a=true,Q=new Z.tB(t.f,s.length,o),L=0;L<s.length;L++)A=s[L],o&&n.rn(t.f,A,true),Q.k[L]=256,Q.e[L]=A,Q.a[L]=n.Pl(g.b,L)/g.a,Q.b[L]=n.Ql(g.b,L)/g.a,D[A]=true,t.b[A]=true;for(u=0;u<C.g;u++)t.c[n.Wo(t.f,s[C.D[0][u]],s[C.D[1][u]])]=true;Z.EA(t.d,Q);}}return D;},Z.wA=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M;for(e=Z.Qz(t),i=z(l._6,J,6,t.f.r,15,1),Z.rA(t,i,e),r=0;r<t.f.r;r++)i[r]==2&&(n.eq(t.f,n.Yl(t.f,0,r))||n.eq(t.f,n.Yl(t.f,1,r)))&&(i[r]=3);for(b=0;b<t.d.a.length;b++){for(s=U.wj(t.d,b),o=Z.kB(s),m=s.c,d=new Z.uB(s),g=-1,h=0;h<224&&o.a.length!=0;h++){if(Q=U.vvb(t.g,o.a.length),u=(w.Kzb(Q,o.a.length),o.a[Q]),D=Z.fA(t,u[0],u[1]),A=z(l.d7,H,6,D.length,15,1),P=0,h<32)for(c=1;c<D.length-1;c++)i[D[c]]==3&&(A[P++]=D[c]);else if(h<96)for(c=1;c<D.length-1;c++)i[D[c]]>=2&&(A[P++]=D[c]);else for(c=1;c<D.length-1;c++)i[D[c]]>=1&&(A[P++]=D[c]);if(P!=0){if(v=A[0],P>1)do v=A[U.vvb(t.g,P)];while(v==g);v!=g&&(g=v,Z.hB(s,v),o=Z.kB(s),m>s.c&&(m=s.c,d=new Z.uB(s)));}}U.Gj(t.d,b,d),s=d,L=1;do{for(C=9999,c=0;c<s.e.length;c++)M=e[s.e[c]],M==L?Z.qB(s,c):M>L&&M<C&&(C=M);L=C;}while(C!=9999);}},Z.xA=function(t,e){t.g=new U.zvb(e);},Z.yA=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v;for(r=z(l.b7,jt,6,n.Ko(t.f,A)+1,15,1),D=z(l.d7,H,6,n.Ko(t.f,A)+1,15,1),u=z(l.d7,H,6,n.Ko(t.f,A)+1,15,1),d=Z.mB(e,A),i=0,Q=0;Q<n.Ko(t.f,A);Q++)D[i]=n.Yo(t.f,A,Q),u[i]=n.$o(t.f,A,Q),s=Z.mB(e,D[i]),s!=-1&&(r[i++]=Z.ZA(e.a[d],e.b[d],e.a[s],e.b[s]));if(i==1)return r[0]+n1;for(L=i-1;L>0;L--)for(h=0;h<L;h++)r[h]>r[h+1]&&(m=r[h],r[h]=r[h+1],r[h+1]=m,C=D[h],D[h]=D[h+1],D[h+1]=C,v=u[h],u[h]=u[h+1],u[h+1]=v);for(r[i]=r[0]+X1,D[i]=D[0],u[i]=u[0],b=-100,c=0,o=0;o<i;o++)P=r[o+1]-r[o],i>2&&n.fq(t.f,u[o])&&n.fq(t.f,u[o+1])&&(g=Z.hA(t,D[o],A,D[o+1]),g!=0&&(P-=100-g)),b<P&&(b=P,c=o);return (r[c]+r[c+1])/2;},Z.zA=function(t,e,A,P){var r;r=j.Math.min(U.yj(t.d,e,0),U.yj(t.d,A,0)),U.qj(t.d,r,P),U.Ej(t.d,e),U.Ej(t.d,A);},Z.AA=function(){Z.BA.call(this,2);},Z.BA=function(t){this.e=t,(t&1)==0&&!Z.Nz&&!Z.Nz&&(Z.Nz=new Z.cB());},Z.CA=function(t,e){var A,P,r,i,D;for(D=0,A=0,P=0;P<e;P++)D+=t[P].b*j.Math.sin(t[P].a),A+=t[P].b*j.Math.cos(t[P].a);return A==0?i=D>0?m0:uA:(i=j.Math.atan(D/A),A<0&&(i+=n1)),r=j.Math.sqrt(D*D+A*A)/e,new Z.XA(i,r);},Z.DA=function(t,e){var A,P;return A=j.Math.abs(t.b),P=j.Math.abs(e.b),A<P?-1:A>P?1:0;},x(68,1,{},Z.AA,Z.BA),y.a=false,y.e=0,l.v8=X(68),Z.EA=function(t,e){var A,P;for(P=new U.wqb(t);P.a<P.c.a.length;)if(A=U.vqb(P),Z.fB(A,e))return  false;return w.nzb(t.a,e),true;},Z.FA=function(t,e){return Z.EA(t,e);},Z.GA=function(){U.Lj.call(this);},x(308,20,WA,Z.GA),y.add=function(t){return Z.FA(this,t);},l.r8=X(308),Z.IA=function(t){var e,A;return e=t.c.e.length,A=t.c.e.length,A<e?-1:A>e?1:0;},Z.JA=function(){},x(310,1,{},Z.JA),y.Qb=function(t,e){var A;return Z.IA((A=t,A));},y.lb=function(t){return this===t;},l.s8=X(310),Z.MA=function(t){var e,A;return e=t.c.e.length,A=t.c.e.length,e<A?-1:e>A?1:0;},Z.NA=function(){},x(311,1,{},Z.NA),y.Qb=function(t,e){var A;return Z.MA((A=t,A));},y.lb=function(t){return this===t;},l.t8=X(311),Z.QA=function(){},x(309,1,{},Z.QA),y.Qb=function(t,e){return Z.DA(t,e);},y.lb=function(t){return this===t;},l.u8=X(309),Z.TA=function(t,e,A){t.c[0]+=Z.iB(t.b[0],e),t.d[0]+=Z.jB(t.b[0],e),t.c[1]+=Z.iB(t.b[1],A),t.d[1]+=Z.jB(t.b[1],A),++t.a[0],++t.a[1];},Z.UA=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V;for(P=z(l.b7,jt,6,2,15,1),Q=0;Q<2;Q++)t.c[Q]/=t.a[Q],t.d[Q]/=t.a[Q],P[Q]=Z.dB(t.b[Q],t.c[Q],t.d[Q],t.b[1-Q].e.length,e);for(Z.rB(t.b[0],t.c[0],t.d[0],m0-P[0]),Z.rB(t.b[1],t.c[1],t.d[1],4.71238898038469-P[1]),V=L0,S=or,D=t.d[0]-t.d[1],L=0;L<t.b[1].b.length;L++)t.b[1].b[L]+=D,V>t.b[1].b[L]&&(V=t.b[1].b[L]),S<t.b[1].b[L]&&(S=t.b[1].b[L]);for(v=S-V+2*e,r=Ft(j.Math.ceil(v)),V+=(v-r)/2-e,m=z(l.b7,jt,6,r,15,1),s=0;s<r;s++)m[s]=t.c[1]+e;for(h=0;h<t.b[1].b.length;h++)for(M=t.b[1].b[h]-V,C=Ft(M-e),u=j.Math.min(Ft(M+e),r-1),d=C;d<=u;d++)m[d]>t.b[1].a[h]&&(m[d]=t.b[1].a[h]);for(b=0;b<r;b++)m[b]-=e;for(i=t.c[0]-t.c[1],c=0;c<t.b[0].a.length;c++)g=Ft(t.b[0].b[c]-V),g>=0&&g<m.length&&i<t.b[0].a[c]-m[g]&&(i=t.b[0].a[c]-m[g]);for(o=0;o<t.b[1].a.length;o++)t.b[1].a[o]+=i;A&&(Z.rB(t.b[0],t.c[0],t.d[0],P[0]-m0),Z.rB(t.b[1],t.c[0],t.d[0],P[0]-m0));},Z.VA=function(t,e){var A,P;for(this.b=z(l.B8,C1,47,2,0,1),this.b[0]=t,this.b[1]=e,this.c=z(l.b7,jt,6,2,15,1),this.d=z(l.b7,jt,6,2,15,1),this.a=z(l.d7,H,6,2,15,1),A=0;A<2;A++){for(P=0;P<this.b[A].e.length;P++)this.c[A]+=Z.iB(this.b[A],P),this.d[A]+=Z.jB(this.b[A],P);this.a[A]=this.b[A].e.length;}},Z.WA=function(t,e,A,P){this.b=z(l.B8,C1,47,2,0,1),this.b[0]=t,this.b[1]=e,this.c=z(l.b7,jt,6,2,15,1),this.d=z(l.b7,jt,6,2,15,1),this.c[0]=Z.iB(this.b[0],A),this.d[0]=Z.jB(this.b[0],A),this.c[1]=Z.iB(this.b[1],P),this.d[1]=Z.jB(this.b[1],P),this.a=z(l.d7,H,6,2,15,1),this.a[0]=1,this.a[1]=1;},x(106,1,{106:1},Z.VA,Z.WA),l.w8=X(106),Z.XA=function(t,e){this.a=t,this.b=e;},Z.YA=function(t,e,A,P){var r,i;this.a=Z.ZA(t,e,A,P),r=A-t,i=P-e,this.b=j.Math.sqrt(r*r+i*i);},Z.ZA=function(t,e,A,P){var r,i,D;return i=A-t,D=P-e,D!=0?(r=j.Math.atan(i/D),D<0&&(i<0?r-=n1:r+=n1)):r=i>0?m0:uA,r;},x(41,1,{41:1},Z.XA,Z.YA),y.a=0,y.b=0,l.x8=X(41),Z.$A=function(t){this.a=z(l.d7,H,6,t,15,1),this.b=z(l.d7,H,6,t,15,1);},x(247,1,{},Z.$A),l.y8=X(247),Z._A=function(t,e,A){this.c=t,this.a=e,this.b=A;},x(197,1,{197:1},Z._A),y.a=0,y.b=0,l.z8=X(197),Z.bB=function(){Z.bB=o1,Z.aB=F(k(l.Sbb,1),st,2,6,["gkvt@@@@LddTTTrbTRTRTRRRRRRRRRRRRRrVRrIh\\IAaQxlY@gRHdJCJcRXlv_CfJx|A\\hRHejiLaQjTje^kSjtFcIhvXmVKMjt{lN{Kavy\\^wGjjjjjjjjjjjjjjjjjjjjjjjjjjjjjjh@@vo@HBC@PhLN@bPhtFKCcpDbILaRhtzCIbsX\\nOO`JDbqDjSKdJeJmQjtz}Ahr[LVkMnpz\\nwGj{PBhBdBlBBBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajijej` !BnkjyVwsVr|iQn|Q|goTZWPIJwbudnRkVYBez]siZymNJZUqNFBqZWxS~iCXVU]SeRjwrtSPAjkvXLpBAZauDPzq]PfMlecrMnkv|@\\SFD`m|mWiEoCXp`SIe_J[l|[XCbloTV`[Gc@FJGopyyoOlFQfUy^w\\Bgz|","gcrt@@@@LdbbbbTRbRbRbRRRRRRRRRRRRVRrVQIA`HtRGAaIxZAHfShTjCIbqylQGKgqdBaXeQJeruBiPitZmFoPZLFSYbvZlVGMnsZ]vWSmr{]UUUUUUUUUUUUUUUUUUUUUUUUUUUUUT@@[G`DAA`HTFG@QHTZCEaqxBQDfPiTZ]AdqYlNWGgpEBQXbUIerEReVhuZ]^`tYMfKUfwX]NW[jkPBhBdBlBbBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajij` !B^cR]`]Fm]QkfljE\\p\x7FUVfgOmFXsQe_gXPyXis_wF|vUUX_XbxpzU]HUFgYViwFo~@uemc@}~T\x7FIEPioYVwr]JnM~[ZEC\\g}~o_pUfdo~irsklTLiyVJshnw^iVAsZ`_~}PYkckURH{FYMImFaRaccUlCZSHMfP","dml@@Dje^VGiyZjjjh@vtHSBinFU@ !BPTCTy[skMzUPF`AJbBixEZHS[Il","dml@@DjYVvGiyZjjjh@vtHSBinFU@ !BwLo~BJ~UquhXBinZ\\ykA@F_eMrT","dml@@LdfbbQX^fUZjjj`C[PaLJfxYT !BzxIHVc{OiJVRpprePho~]}y\x7FwLl","deL@@DjUYkfEijjjj@MeBDpj[ad !B\x7FaA\x7FMVr[AvkKzm_jKvVbD{sk","dil@@LddTQRl[NX^Fjjjj@MiBDpj[a@ !BPfL@\x7Fox@M~T@\x7Fox@\x7F`C~@@","daL@@DjYtKJqjynjjjj@MaBDpj[` !B`bL@_gx@@Gy~@Gx@_`@"]);},Z.cB=function(){Z.bB();var t,e,A,P,r,i,D;for(U.Lj.call(this),i=new n.xu(),A=Z.aB,P=0,r=A.length;P<r;++P)e=A[P],t=n.Oq(new n.Wq(),e),n.qu(i,t),D=new Z.vB(t),D.a=n.Tl(D.b),w.nzb(this.a,D);},x(362,20,WA,Z.cB),l.A8=X(362),Z.dB=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt;if(t.e.length==1)return 0;for(N=r+j.Math.sqrt(P),L=z(l.b7,jt,6,36,15,1),c=0;c<t.e.length;c++)i=Z.ZA(e,A,t.a[c],t.b[c]),u=Z.eB(Gt(w1(j.Math.round(i*36/X1)))),s=e-t.a[c],h=A-t.b[c],ut=s*s+h*h,L[u]<ut&&(L[u]=ut);for(S=-1,M=-1,g=0;g<36;g++)L[g]=j.Math.sqrt(L[g]),S<L[g]&&(S=L[g],M=g);for(I=M-18<0?M-18+36:M-18>=36?M-18-36:M-18,d=0;d<=18;d++)L[I+d<0?I+d+36:I+d>=36?I+d-36:I+d]+=.01*d,L[I-d<0?I-d+36:I-d>=36?I-d-36:I-d]+=.01*d;for(Qt=z(l.b7,jt,6,9,15,1),o=z(l.b7,jt,6,9,15,1),m=1;m<9;m++)Qt[m]=j.Math.sin(m*Pi),o[m]=j.Math.cos(m*Pi);for(O=L0,V=-1,D=0;D<36;D++)if(!(L[D]>=O)){for(v=L[D],b=1;b<9;b++){for(C=-1;C<=1&&(K=D+C*b<0?D+C*b+36:D+C*b>=36?D+C*b-36:D+C*b,!(!(L[K]*o[b]<=v)&&(Q=o[b]*j.Math.min(L[K],N/Qt[b]),v<Q&&(v=Q,O<=v))));C+=2);if(O<=v)break;}O>v&&(O=v,V=D);}return X1*V/36;},Z.eB=function(t){return t<0?t+36:t>=36?t-36:t;},Z.fB=function(t,e){var A,P,r;if(e.e.length!=t.e.length)return  false;for(P=Z.nB(t),r=Z.nB(e),A=0;A<P.length;A++)if(P[A]!=r[A])return  false;return  true;},Z.gB=function(t,e,A){var P,r,i;for(i=0;i<t.e.length;i++)r=j.Math.sqrt((t.a[i]-e)*(t.a[i]-e)+(t.b[i]-A)*(t.b[i]-A)),P=0-Z.ZA(e,A,t.a[i],t.b[i]),t.a[i]=e+r*j.Math.sin(P),t.b[i]=A+r*j.Math.cos(P);},Z.hB=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V;if(t.d==null&&(t.d=z(l.d7,S1,7,t.j.r,0,2)),t.d[e]==null){for(b=z(l.d7,H,6,t.e.length,15,1),v=z(l.zeb,ht,6,t.j.q,16,1),r=n.Yl(t.j,0,e),i=n.Yl(t.j,1,e),b[0]=r,v[r]=true,L=0,c=0;L<=c;){for(d=0;d<n.Ko(t.j,b[L]);d++)D=n.Yo(t.j,b[L],d),!v[D]&&D!=i&&(b[++c]=D,v[D]=true);if(L==c)break;++L;}if(h=c+1>(t.e.length/2|0),t.i){for(o=false,u=false,d=0;d<t.e.length;d++)P=t.e[d],n.Jm(t.j,P)&&P!=r&&P!=i&&(v[t.e[d]]?o=true:u=true);o!=u&&(h=o);}for(Q=2,t.d[e]=z(l.d7,H,6,h?t.e.length-c:c+2,15,1),m=0;m<t.e.length;m++)t.e[m]==r?t.d[e][h?0:1]=m:t.e[m]==i?t.d[e][h?1:0]=m:h^v[t.e[m]]&&(t.d[e][Q++]=m);}for(S=t.a[t.d[e][0]],V=t.b[t.d[e][0]],M=Z.ZA(S,V,t.a[t.d[e][1]],t.b[t.d[e][1]]),g=2;g<t.d[e].length;g++)C=t.d[e][g],s=j.Math.sqrt((t.a[C]-S)*(t.a[C]-S)+(t.b[C]-V)*(t.b[C]-V)),A=2*M-Z.ZA(S,V,t.a[C],t.b[C]),t.a[C]=S+s*j.Math.sin(A),t.b[C]=V+s*j.Math.cos(A);},Z.iB=function(t,e){return t.a[e];},Z.jB=function(t,e){return t.b[e];},Z.kB=function(t){var e,A,P,r,i,D,u,o;for(t.c=0,A=new U.Lj(),r=1;r<t.e.length;r++)for(i=0;i<r;i++)u=j.Math.abs(t.a[r]-t.a[i]),o=j.Math.abs(t.b[r]-t.b[i]),P=j.Math.sqrt(u*u+o*o),P<.8&&(e=z(l.d7,H,6,2,15,1),e[0]=t.e[r],e[1]=t.e[i],w.nzb(A.a,e)),D=1-j.Math.min(P,1),t.c+=D*D;return A;},Z.lB=function(t,e){return t.e[e];},Z.mB=function(t,e){var A;for(A=0;A<t.e.length;A++)if(e==t.e[A])return A;return  -1;},Z.nB=function(t){return t.n==null&&(t.n=U.Eqb(t.e,t.e.length),w.qzb(t.n,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[])))),t.n;},Z.oB=function(t,e){var A;for(A=0;A<t.e.length;A++)if(e==t.e[A])return  true;return  false;},Z.pB=function(t){var e,A,P,r,i,D;for(P=0,i=0;i<t.e.length;i++)for(e=t.e[i],A=n.Ko(t.j,e),D=0;D<A;D++)n.Yo(t.j,e,D)>e&&++P;for(t.f=z(l.d7,H,6,P,15,1),t.g=z(l.d7,H,6,t.j.q,15,1),P=0,r=0;r<t.e.length;r++)for(e=t.e[r],A=n.Ko(t.j,e),t.g[e]=r,D=0;D<A;D++)n.Yo(t.j,e,D)>e&&(t.f[P++]=n.$o(t.j,e,D));},Z.qB=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;for(c=t.a[e],C=t.b[e],i=z(l.x8,C1,41,4,0,1),L=0,s=0;s<t.f.length&&!(L>=4);s++)if(!(e==t.g[n.Yl(t.j,0,t.f[s])]||e==t.g[n.Yl(t.j,1,t.f[s])])){if(g=t.a[t.g[n.Yl(t.j,0,t.f[s])]],v=t.b[t.g[n.Yl(t.j,0,t.f[s])]],d=t.a[t.g[n.Yl(t.j,1,t.f[s])]],M=t.b[t.g[n.Yl(t.j,1,t.f[s])]],u=j.Math.sqrt((g-c)*(g-c)+(v-C)*(v-C)),o=j.Math.sqrt((d-c)*(d-c)+(M-C)*(M-C)),r=j.Math.sqrt((d-g)*(d-g)+(M-v)*(M-v)),u<r&&o<r){g==d?(D=j.Math.abs(c-g),D<.5&&(i[L++]=new Z.XA(Z.ZA(g,C,c,C),(.5-D)/2))):v==M?(D=j.Math.abs(C-v),D<.5&&(i[L++]=new Z.XA(Z.ZA(c,v,c,C),(.5-D)/2))):(h=(M-v)/(d-g),b=-1/h,A=v-h*g,P=C-b*c,m=(P-A)/(h-b),S=h*m+A,D=j.Math.sqrt((m-c)*(m-c)+(S-C)*(S-C)),D<.5&&(i[L++]=new Z.XA(Z.ZA(m,S,c,C),(.5-D)/2)));continue;}if(u<.5){i[L++]=new Z.XA(Z.ZA(g,v,c,C),(.5-u)/2);continue;}if(o<.5){i[L++]=new Z.XA(Z.ZA(d,M,c,C),(.5-o)/2);continue;}}L>0&&(Q=Z.CA(i,L),t.a[e]+=Q.b*j.Math.sin(Q.a),t.b[e]+=Q.b*j.Math.cos(Q.a));},Z.rB=function(t,e,A,P){var r,i,D;for(D=0;D<t.e.length;D++)i=j.Math.sqrt((t.a[D]-e)*(t.a[D]-e)+(t.b[D]-A)*(t.b[D]-A)),r=Z.ZA(e,A,t.a[D],t.b[D])+P,t.a[D]=e+i*j.Math.sin(r),t.b[D]=A+i*j.Math.cos(r);},Z.sB=function(t,e,A){var P;for(P=0;P<t.e.length;P++)t.a[P]+=e,t.b[P]+=A;},Z.tB=function(t,e,A){this.j=t,this.i=A,this.e=z(l.d7,H,6,e,15,1),this.k=z(l.d7,H,6,e,15,1),this.a=z(l.b7,jt,6,e,15,1),this.b=z(l.b7,jt,6,e,15,1);},Z.uB=function(t){var e,A;for(this.j=t.j,this.i=t.i,this.e=z(l.d7,H,6,t.e.length,15,1),this.k=z(l.d7,H,6,t.e.length,15,1),this.a=z(l.b7,jt,6,t.e.length,15,1),this.b=z(l.b7,jt,6,t.e.length,15,1),A=0;A<t.e.length;A++)this.e[A]=t.e[A],this.k[A]=t.k[A],this.a[A]=t.a[A],this.b[A]=t.b[A];if(t.f!=null)for(this.f=z(l.d7,H,6,t.f.length,15,1),e=0;e<t.f.length;e++)this.f[e]=t.f[e];if(t.g!=null)for(this.g=z(l.d7,H,6,t.g.length,15,1),e=0;e<t.g.length;e++)this.g[e]=t.g[e];},x(47,1,{47:1},Z.tB,Z.uB),y.c=0,y.i=false,l.B8=X(47),Z.vB=function(t){this.b=t,this.c=false;},x(246,1,{246:1},Z.vB),y.a=0,y.c=false,l.C8=X(246),we.wB=function(t){t.c=new U.Lj();},we.xB=function(t,e,A,P,r,i){var D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut;for(v=-1,O=0,V=0,N=0,d=0,m=0,C=0,I=0,ut=0,O=0,L=0;L<t.d;L++)O+=P[L]*P[L];if(O=j.Math.sqrt(O),O>i)for(s=0;s<t.d;s++)P[s]*=i/O;for(V=0,h=0;h<t.d;h++)V+=P[h]*t.e[h];if(V>=0)return v;for(N=0,b=0;b<t.d;b++)K=j.Math.abs(P[b])/j.Math.max(j.Math.abs(e[b]),1),K>N&&(N=K);for(C=1e-7/N,d=1,g=0;g<1e3;){if(d<C){v=1;break;}for(c=0;c<t.d;c++)r[c]=e[c]+d*P[c];if(t.k=Y.wC(t,r),t.k-A<=wA*d*V)return 0;g==0?I=-V/(2*(t.k-A-V)):(M=t.k-A-d*V,S=ut-A-m*V,D=(M/(d*d)-S/(m*m))/(d-m),u=(-m*M/(d*d)+d*S/(m*m))/(d-m),D==0?I=-V/(2*u):(o=u*u-3*D*V,o<0?I=.5*d:u<=0?I=(-u+j.Math.sqrt(o))/(3*D):I=-V/(u+j.Math.sqrt(o))),I>.5*d&&(I=.5*d)),m=d,ut=t.k,d=j.Math.max(I,.1*d),++g;}for(Q=0;Q<t.d;Q++)r[Q]=e[Q];return v;},we.yB=function(t,e,A){var P,r,i,D;for(D=1,r=0;r<t.g.q;r++)t.i[3*r]=n.Pl(t.g,r),t.i[3*r+1]=n.Ql(t.g,r),t.i[3*r+2]=n.Rl(t.g,r);if(D=we.zB(t,e,A),D==0)for(P=0;P<t.g.q;P++)n.yn(t.g,P,t.i[3*P]),n.zn(t.g,P,t.i[3*P+1]),n.An(t.g,P,t.i[3*P+2]);for(i=new U.wqb(t.c);i.a<i.c.a.length;)U.vqb(i),null.xd();return D;},we.zB=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct;for(t.e=z(l.b7,jt,6,t.d,15,1),P=z(l.b7,jt,6,t.d,15,1),Q=z(l.b7,jt,6,t.d,15,1),K=z(l.b7,jt,6,t.d,15,1),ct=z(l.b7,jt,6,t.d,15,1),C=z(l.b7,jt,6,t.d*t.d,15,1),s=0;s<t.d;s++)K[s]=t.i[s];for(u=Y.wC(t,t.i),Y.xC(t),I=0,h=0;h<t.d;h++)C[h*t.d+h]=1,ct[h]=-t.e[h],I+=t.i[h]*t.i[h];for(O=100*j.Math.max(j.Math.sqrt(I),t.d),Dt=(T.Ujb(),w1(Date.now())),M=1;M<=e&&!t.f;M++){if(N=we.xB(t,t.i,u,ct,K,O),N<0)return 2;for(u=t.k,yt=0,b=0;b<t.d;b++)ct[b]=K[b]-t.i[b],t.i[b]=K[b],ot=j.Math.abs(ct[b])/j.Math.max(j.Math.abs(t.i[b]),1),ot>yt&&(yt=ot),P[b]=t.e[b];if(yt<12e-8)return 0;for(o=Y.xC(t),yt=0,rt=j.Math.max(t.k*o,1),c=0;c<t.d;c++)it=j.Math.abs(t.e[c])*j.Math.max(j.Math.abs(t.i[c]),1),yt=j.Math.max(yt,it),P[c]=t.e[c]-P[c];if(yt/=rt,yt<A)return 0;for(r=0,D=0,ut=0,Qt=0,g=0;g<t.d;g++){for(v=g*t.d,Q[g]=0,S=0;S<t.d;S++)Q[g]+=C[v+S]*P[S];r+=P[g]*ct[g],D+=P[g]*Q[g],ut+=P[g]*P[g],Qt+=ct[g]*ct[g];}if(r>j.Math.sqrt(3e-8*ut*Qt)){for(r=1/r,i=1/D,d=0;d<t.d;d++)P[d]=r*ct[d]-i*Q[d];for(m=0;m<t.d;m++)for(v=m*t.d,S=m;S<t.d;S++)C[v+S]+=r*ct[m]*ct[S]-i*Q[m]*Q[S]+D*P[m]*P[S],C[S*t.d+m]=C[v+S];}for(L=0;L<t.d;L++)for(v=L*t.d,ct[L]=0,S=0;S<t.d;S++)ct[L]-=C[v+S]*t.e[S];if(Lt=w1(Date.now()),bt=O1(Lt,Dt),zn(bt,t.j)){for(V=new U.wqb(t.c);V.a<V.c.a.length;)U.vqb(V),null.xd();Dt=Lt;}}return 1;},we.AB=function(t){var e,A,P;for(we.wB(this),P=0,e=0;e<t.f;e++)P+=n.np(t,e);if(P>0)throw Ut(new T.dib("molecule needs explicit hydrogen atoms for force field calculations"));for(this.g=t,this.d=3*t.q,this.e=z(l.b7,jt,6,this.d,15,1),this.i=z(l.b7,jt,6,this.d,15,1),this.f=false,this.j=20,A=0;A<t.q;A++)this.i[3*A]=t.J[A].a,this.i[3*A+1]=t.J[A].b,this.i[3*A+2]=t.J[A].c;},x(275,1,{}),y.d=0,y.f=false,y.j=0,y.k=0,l.D8=X(275),Y.BB=function(t,e,A,P,r){this.f=e,this.a=A,this.b=P,this.c=r,this.d=At.aE(t.b,e.a[P]),this.g=At.VD(t.a,e,A,P,r),this.e=At.UD(t.a,e,A,P,r);},Y.CB=function(t,e){var A,P,r,i,D,u;for(A=new U.Lj(),P=0;P<e.q;P++)if(e.e[P]>1)for(r=0;r<e.e[P];r++)for(D=e.i[P][r],i=r+1;i<e.e[P];i++)u=e.i[P][i],U.rj(A,new Y.BB(t,e,D,P,u));return A;},x(353,1,Un,Y.BB),y.bc=function(t,e,A,P){var r,i,D;return P&&(n.Sl(this.f,this.a)==1||n.Sl(this.f,this.c)==1)?0:(D=Y.CD(new Y.OD(t,this.b,this.a),new Y.OD(t,this.b,this.c)),r=D*sn-this.g,i=this.d?143.9325*this.e*(1+j.Math.cos(D)):ri*this.e*r*r*(1+-6981317e-9*r),i);},y.cc=function(t,e){var A,P,r,i,D,u,o,Q,L,s;o=Y.HD(new Y.OD(t,this.b,this.a)),Q=Y.HD(new Y.OD(t,this.b,this.c)),D=Y.GD(new Y.OD(t,this.b,this.a)),u=Y.GD(new Y.OD(t,this.b,this.c)),P=(o.a*Q.a+o.b*Q.b+o.c*Q.c)/(j.Math.sqrt(o.a*o.a+o.b*o.b+o.c*o.c)*j.Math.sqrt(Q.a*Q.a+Q.b*Q.b+Q.c*Q.c)),s=1-P*P,L=gA,s>0&&(L=j.Math.sqrt(s)),A=sn*j.Math.acos(P)-this.g,i=De*this.e*A*(1+-0.0104719755*A),this.d&&(i=-143.9325*this.e*L),r=F(k(l.b7,1),jt,6,15,[1/D*(Q.a-P*o.a),1/D*(Q.b-P*o.b),1/D*(Q.c-P*o.c),1/u*(o.a-P*Q.a),1/u*(o.b-P*Q.b),1/u*(o.c-P*Q.c)]),e[3*this.a]+=i*r[0]/-L,e[3*this.a+1]+=i*r[1]/-L,e[3*this.a+2]+=i*r[2]/-L,e[3*this.b]+=i*(-r[0]-r[3])/-L,e[3*this.b+1]+=i*(-r[1]-r[4])/-L,e[3*this.b+2]+=i*(-r[2]-r[5])/-L,e[3*this.c]+=i*r[3]/-L,e[3*this.c+1]+=i*r[4]/-L,e[3*this.c+2]+=i*r[5]/-L;},y.a=0,y.b=0,y.c=0,y.d=false,y.e=0,y.g=0,l.E8=X(353),Y.fC=function(t){T.eC.call(this,t);},x(185,33,{185:1,4:1,19:1,33:1,27:1},Y.fC),l.F8=X(185),Y.gC=function(){T.cC.call(this);},x(315,33,B0,Y.gC),l.G8=X(315),Y.hC=function(t,e,A){Y.iC.call(this,t,e,e.D[0][A],e.D[1][A]);},Y.iC=function(t,e,A,P){this.d=e,this.a=A,this.b=P,this.e=At.oE(t.d,e,A,P),this.c=At.nE(t.d,e,A,P);},Y.jC=function(t,e){var A,P;for(P=new U.Lj(),A=0;A<e.r;A++)U.rj(P,new Y.hC(t,e,A));return P;},x(237,1,Un,Y.hC),y.bc=function(t,e,A,P){var r,i,D;return P&&(n.Sl(this.d,this.a)==1||n.Sl(this.d,this.b)==1)?0:(i=Y.ED(new Y.ND(t,this.a),new Y.ND(t,this.b)),r=(i-this.e)*(i-this.e),D=71.96625*this.c*r*(1+-2*(i-this.e)+2.3333333333333335*r),D);},y.cc=function(t,e){var A,P,r,i;if(P=Y.ED(new Y.ND(t,this.a),new Y.ND(t,this.b)),r=P-this.e,A=143.9325*this.c*r*(1+-3*r+4.666666666666667*r*r),P>0)for(i=0;i<3;i++)e[3*this.a+i]+=A*(t[3*this.a+i]-t[3*this.b+i])/P,e[3*this.b+i]-=A*(t[3*this.a+i]-t[3*this.b+i])/P;},y.a=0,y.b=0,y.c=0,y.e=0,l.H8=X(237),Y.mC=function(t){var e,A,P,r,i,D,u,o,Q;e=null;try{for(e=new Rt.igb(new Rt.ngb((r=_t.hAb(t),V1.Hmb(),r))),o=T.Qhb(T.wjb(Rt.hgb(e)),P1,St),P=T.sjb(T.wjb(Rt.hgb(e)),","),Q=h1(l.Mbb,[c0,C1],[8,1],5,[o,P.length],2),D=0;(i=Rt.hgb(e))!=null&&D<o;D++)for(u=T.sjb(T.wjb(i),","),A=0;A<P.length;A++)switch(T.Yib(P[A],0)){case 105:a1(Q[D],A,T.uib(T.Qhb(T.wjb(u[A]),P1,St)));break;case 102:a1(Q[D],A,w.Xzb(T.Phb(T.wjb(u[A]))));break;case 99:a1(Q[D],A,P2(T.Yib(T.qjb(T.qjb(T.wjb(u[A]),"'"),'"'),0)));}return Q;}catch(L){if(L=H1(L),Jt(L,76))T.Ujb();else throw Ut(L);}return h1(l.Mbb,[c0,C1],[8,1],5,[1,1],2);},Y.nC=function(t){var e,A,P,r,i,D;for(D=Y.mC(t),i=D.length,e=D[0].length,r=h1(l.d7,[S1,H],[7,6],15,[i,e],2),A=0;A<i;A++)for(P=0;P<e;P++)r[A][P]=D[A][P].a;return r;},Y.oC=function(t,e,A,P,r,i,D,u){this.e=t,this.a=e,this.b=A,this.f=P,this.d=D,this.c=r*i/u;},Y.pC=function(t,e,A,P,r,i){var D,u,o,Q,L;for(u=new U.Lj(),D=i1.BF(t,e),o=0;o<e.q;o++)for(Q=0;Q<o+1;Q++)L=Y.YC(A,new Y.hD(o,Q)),(L==(Y.dD(),Y.$C)||L==Y.cD)&&j.Math.abs(D[o])>zA&&j.Math.abs(D[Q])>zA&&Y.GD(new Y.LD(e,o,Q))<P&&U.rj(u,new Y.oC(e,o,Q,L,D[o],D[Q],r,i));return u;},x(354,1,Un,Y.oC),y.bc=function(t,e,A,P){var r,i,D;return P&&(n.Sl(this.e,this.a)==1||n.Sl(this.e,this.b)==1)?0:(i=Y.GD(new Y.OD(t,this.a,this.b)),r=i+.05,this.d&&(r*=r),D=332.0716*this.c/r*(this.f==(Y.dD(),Y.$C)?.75:1),D);},y.cc=function(t,e){var A,P,r,i,D;for(i=Y.GD(new Y.OD(t,this.a,this.b)),A=i+.05,A*=this.d?A*A:A,P=-332.0716*(this.d?2:1)*this.c/A*(this.f==(Y.dD(),Y.$C)?.75:1),D=0;D<3;D++)r=.02,i>0&&(r=P*(t[3*this.a+D]-t[3*this.b+D])/i),e[3*this.a+D]+=r,e[3*this.b+D]-=r;},y.a=0,y.b=0,y.c=0,y.d=false,l.I8=X(354),Y.tC=function(){Y.tC=o1,Y.sC=new U.utb();},Y.uC=function(t){t.a=new U.Lj();},Y.vC=function(t){return Y.wC(t,t.i);},Y.wC=function(t,e){var A,P,r,i,D;for(D=0,P=1,i=new U.wqb(t.a);i.a<i.c.a.length;)r=U.vqb(i),A=r.bc(e,null,""+P,false),A!=0&&(D+=A,++P);return D;},Y.xC=function(t){var e,A,P,r,i,D;for(t.e=z(l.b7,jt,6,t.d,15,1),A=new U.wqb(t.a);A.a<A.c.a.length;)e=U.vqb(A),e.cc(t.i,t.e);for(D=-1e8,P=.1,i=0;i<t.d;i++)t.e[i]*=P,t.e[i]>D&&(D=t.e[i]);if(D>10){for(;D*P>10;)P*=.5;for(r=0;r<t.d;r++)t.e[r]*=P;}return P;},Y.yC=function(t){Y.tC(),Y.zC.call(this,t,sP,new U.utb());},Y.zC=function(t,e,A){Y.tC();var P,r,i,D,u;we.AB.call(this,t),Y.uC(this),this.b=new Y.BC(t),n.aw(this.g,7),u=U.tnb(Y.sC,e),i=U.aub(A.c,ii)?T.Vhb(U.bub(A.c,ii)):100,P=U.aub(A.c,fi)?T.Vhb(U.bub(A.c,fi)):1,r=U.aub(A.c,Di)&&T.Oc(U.bub(A.c,Di),"distance"),D=new Y.ZC(this.b),(!U.aub(A.c,ui)||T.Lgb(U.bub(A.c,ui)))&&U.tj(this.a,Y.CB(u,this.b)),(!U.aub(A.c,oi)||T.Lgb(U.bub(A.c,oi)))&&U.tj(this.a,Y.jC(u,this.b)),(!U.aub(A.c,li)||T.Lgb(U.bub(A.c,li)))&&U.tj(this.a,Y.pC(u,this.b,D,i,r,P)),(!U.aub(A.c,Qi)||T.Lgb(U.bub(A.c,Qi)))&&U.tj(this.a,Y.DC(u,this.b)),(!U.aub(A.c,Li)||T.Lgb(U.bub(A.c,Li)))&&U.tj(this.a,Y.nD(u,this.b)),(!U.aub(A.c,yi)||T.Lgb(U.bub(A.c,yi)))&&U.tj(this.a,Y.tD(u,this.b)),(!U.aub(A.c,Ui)||T.Lgb(U.bub(A.c,Ui)))&&U.tj(this.a,Y.zD(u,this.b,D,i));},Y.AC=function(t,e){Y.tC(),U.unb(Y.sC,t)||U.wnb(Y.sC,t,e);},x(177,275,{},Y.yC,Y.zC),l.J8=X(177),Y.BC=function(t){n.Qk();var e,A,P,r,i;for(n.uw.call(this,t),n.jp(this,n.Jo(this)),i=(n.aw(this,7),this.p),this.b=z(l.M8,H,6,i.j.a.length,15,1),U.Kqb(this.b),e=false,A=true;!e&&A;)for(e=true,A=false,r=0;r<i.j.a.length;r++)this.b[r]==2&&(this.b[r]=i1.yF(this,r),this.b[r]!=2&&(A=true)),this.b[r]==2&&(e=false);if(!e)throw Ut(new Y.gC());for(this.a=z(l.d7,H,6,this.q,15,1),P=0;P<this.a.length;P++)if(this.a[P]=-1,this.a[P]=i1.sF(this,P),this.a[P]==0)throw Ut(new Y.fC("Couldn't assign an atom type to atom "+P+" ("+n.Kk[this.C[P]]+")"));},x(276,25,f9,Y.BC),l.K8=X(276),Y.CC=function(t,e,A,P,r,i){this.f=e,this.d=A,this.a=P,this.b=r,this.c=i,this.e=At.PE(t.k,e,A,P,r,i);},Y.DC=function(t,e){var A,P,r,i,D;for(D=new U.Lj(),i=0;i<e.q;i++)e.e[i]==3&&(A=e.i[i][0],P=e.i[i][1],r=e.i[i][2],U.rj(D,new Y.CC(t,e,i,A,P,r)),U.rj(D,new Y.CC(t,e,i,A,r,P)),U.rj(D,new Y.CC(t,e,i,P,r,A)));return D;},x(192,1,Un,Y.CC),y.bc=function(t,e,A,P){var r,i,D,u,o,Q;return P&&(n.Sl(this.f,this.a)==1||n.Sl(this.f,this.b)==1||n.Sl(this.f,this.c)==1)?0:(u=Y.HD(new Y.OD(t,this.d,this.a)),o=Y.HD(new Y.OD(t,this.d,this.b)),Q=Y.HD(new Y.OD(t,this.d,this.c)),D=Y.HD(new Y.JD(u.b*o.c-u.c*o.b,u.c*o.a-u.a*o.c,u.a*o.b-u.b*o.a)),r=sn*j.Math.asin(D.a*Q.a+D.b*Q.b+D.c*Q.c),i=ri*this.e*r*r,i);},y.cc=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N;for(h=new Y.OD(t,this.d,this.a),b=new Y.OD(t,this.d,this.b),c=new Y.OD(t,this.d,this.c),u=j.Math.sqrt(h.a*h.a+h.b*h.b+h.c*h.c),o=j.Math.sqrt(b.a*b.a+b.b*b.b+b.c*b.c),Q=j.Math.sqrt(c.a*c.a+c.b*c.b+c.c*c.c),h=Y.HD(h),b=Y.HD(b),c=Y.HD(c),s=Y.HD(Y.DD(new Y.JD(-h.a,-h.b,-h.c),b)),g=c.a*s.a+c.b*s.b+c.c*s.c,r=1-g*g,P=j.Math.max(r>0?j.Math.sqrt(r):0,gA),A=sn*j.Math.asin(g),i=h.a*b.a+h.b*b.b+h.c*b.c,m=j.Math.max(1-i*i,gA),d=j.Math.max(m>0?j.Math.sqrt(m):0,gA),D=De*this.e*A,C=new Y.JD(c.b*b.c-c.c*b.b,c.c*b.a-c.a*b.c,c.a*b.b-c.b*b.a),v=new Y.JD(h.b*c.c-h.c*c.b,h.c*c.a-h.a*c.c,h.a*c.b-h.b*c.a),M=new Y.JD(b.b*h.c-b.c*h.b,b.c*h.a-b.a*h.c,b.a*h.b-b.b*h.a),S=P*d,V=g/(P*m),O=F(k(l.b7,1),jt,6,15,[(C.a/S-(h.a-b.a*i)*V)/u,(C.b/S-(h.b-b.b*i)*V)/u,(C.c/S-(h.c-b.c*i)*V)/u]),K=F(k(l.b7,1),jt,6,15,[(v.a/S-(b.a-h.a*i)*V)/o,(v.b/S-(b.b-h.b*i)*V)/o,(v.c/S-(b.c-h.c*i)*V)/o]),N=F(k(l.b7,1),jt,6,15,[(M.a/S-c.a*g/P)/Q,(M.b/S-c.b*g/P)/Q,(M.c/S-c.c*g/P)/Q]),L=0;L<3;L++)e[3*this.a+L]+=D*O[L],e[3*this.d+L]+=-D*(O[L]+K[L]+N[L]),e[3*this.b+L]+=D*K[L],e[3*this.c+L]+=D*N[L];},y.a=0,y.b=0,y.c=0,y.d=0,y.e=0,l.L8=X(192),Y.GC=function(t){return t<3?0:t<11?1:t<19?2:t<37?3:t<55?4:0;},Y.HC=function(t){var e;return e=0,t==2?e=1:t>=3&&t<=10?e=2:t>=11&&t<=18?e=3:t>=19&&t<=36?e=4:t>=37&&t<=54&&(e=5),(t>=21&&t<=30||t>=39&&t<=48)&&(e*=10),e;},T.IC=function(t,e){return t.b-e.b;},T.JC=function(t){return t.a!=null?t.a:""+t.b;},T.KC=function(t){return t.b;};function FA(f,t){this.a=f,this.b=t;}x(56,1,{4:1,32:1,56:1}),y.Rb=function(t){return T.IC(this,t);},y.compareTo=function(t){return T.IC(this,t);},y.equals=function(t){return this===t;},y.lb=function(f){return this.equals(f);},y.hashCode=function(){return w.Dzb(this);},y.nb=function(){return this.hashCode();},y.name=function(){return T.JC(this);},y.ordinal=function(){return T.KC(this);},y.toString=function(){return this.a!=null?this.a:""+this.b;},y.ob=function(){return this.toString();},y.b=0,l.Abb=X(56),Y.TC=function(){return F(k(l.M8,1),H,6,15,[0,1,2]);},l.M8=Xe(null,Y.TC),Y.UC=function(t,e,A,P,r,i){var D,u,o,Q;if(A=j.Math.max(0,A),P=j.Math.min(i.kc(),P),D=(P-A)/2|0,o=A,u=P,i.jc(o,t)>e||i.jc(u-1,t)<e)return  -1;for(;P>=A;){if(D=A+((P-A)/2|0),Q=i.jc(D,t),Q==e){if(r&&D>o&&i.jc(D-1,t)==e)P=D;else if(!r&&D<u-1&&i.jc(D+1,t)==e)A=D;else return D;}else Q>e?P=D:Q<e&&(A=D);if(P-A==1&&i.jc(A,t)<e&&i.jc(P,t)>e)break;}return  -1;},Y.VC=function(t,e){return Y.UC(0,t,0,e.a.length,true,e);},Y.WC=function(t,e,A){var P,r,i;if(t.length!=e.length)return  -1;for(i=0,P=A.kc(),r=0;r<t.length-1;r++)if(i=Y.UC(t[r],e[r],i,P+1,true,A),P=Y.UC(t[r],e[r],i,P+1,false,A),i==-1||P==-1)return  -1;return Y.UC(t[r],e[r],i,P+1,true,A);},Y.XC=function(t){t.a=new U.utb();},Y.YC=function(t,e){return U.snb(t.a,e)!=null?U.snb(t.a,e):(Y.dD(),Y.cD);},Y.ZC=function(t){var e,A,P,r,i,D,u,o,Q,L;for(Y.XC(this),e=0;e<t.q;e++)for(U.vnb(this.a,new Y.hD(e,e),(Y.dD(),Y._C)),A=0;A<t.e[e];A++)for(o=t.i[e][A],u=new Y.hD(e,o),U.vnb(this.a,u,Y.bD),P=0;P<t.e[o];P++)for(Q=t.i[o][P],D=new Y.hD(e,Q),(!U.onb(this.a,D)||k1(U.snb(this.a,D))===k1(Y.$C))&&U.vnb(this.a,D,Y.aD),r=0;r<t.e[Q];r++)L=t.i[Q][r],i=new Y.hD(e,L),U.onb(this.a,i)||U.vnb(this.a,i,Y.$C);},x(277,1,{},Y.ZC),l.O8=X(277),Y.dD=function(){Y.dD=o1,Y._C=new Y.eD("ONE_ONE",0),Y.bD=new Y.eD("ONE_TWO",1),Y.aD=new Y.eD("ONE_THREE",2),Y.$C=new Y.eD("ONE_FOUR",3),Y.cD=new Y.eD("ONE_X",4);},Y.eD=function(t,e){FA.call(this,t,e);},Y.fD=function(){return Y.dD(),F(k(l.N8,1),s0,108,0,[Y._C,Y.bD,Y.aD,Y.$C,Y.cD]);},x(108,56,{108:1,4:1,32:1,56:1},Y.eD),l.N8=Xe(108,Y.fD),Y.gD=function(t,e){return t.a>e.a?1:t.a<e.a?-1:t.b>e.b?1:t.b<e.b?-1:0;},Y.hD=function(t,e){this.a=t>e?t:e,this.b=t>e?e:t;},x(99,1,{99:1,32:1},Y.hD),y.Rb=function(t){return Y.gD(this,t);},y.lb=function(t){var e;return t===this?true:Jt(t,99)?(e=t,this.a==e.a&&this.b==e.b):false;},y.nb=function(){return this.a^this.b;},y.ob=function(){return this.a+","+this.b;},y.a=0,y.b=0,l.P8=X(99),Y.mD=function(t,e,A,P,r){this.f=e,this.a=A,this.b=P,this.c=r,this.j=At.VD(t.a,e,A,P,r),this.g=At.oE(t.d,e,A,P),this.i=At.oE(t.d,e,r,P),this.d=At.UE(t.n,e,A,P,r),this.e=At.UE(t.n,e,r,P,A);},Y.nD=function(t,e){var A,P,r,i,D,u,o,Q;for(Q=new U.Lj(),A=0;A<e.q;A++)if(P=e.a[A],!(e.e[A]<=1&&At.aE(t.b,P)))for(i=0;i<e.e[A];i++)for(u=e.i[A][i],D=0;D<e.e[A];D++)o=e.i[A][D],!(u>o)&&Y.qD(t,e,u,A,o)&&(r=new Y.mD(t,e,u,A,o),(j.Math.abs(r.d)>wA||j.Math.abs(r.e)>wA)&&w.nzb(Q.a,r));return Q;},Y.qD=function(t,e,A,P,r){var i;return i=e.a[P],!(At.aE(t.b,i)||A==r||n.Wo(e,A,P)==-1||n.Wo(e,P,r)==-1);},x(355,1,Un,Y.mD),y.bc=function(t,e,A,P){var r,i,D,u,o;return P&&(n.Sl(this.f,this.a)==1||n.Sl(this.f,this.c)==1)?0:(r=Y.GD(new Y.OD(t,this.b,this.a)),i=Y.GD(new Y.OD(t,this.b,this.c)),o=Y.CD(new Y.OD(t,this.b,this.a),new Y.OD(t,this.b,this.c)),u=De*(o*sn-this.j),D=u*(r-this.g)*this.d+u*(i-this.i)*this.e,D);},y.cc=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d;L=Y.GD(new Y.OD(t,this.b,this.a)),s=Y.GD(new Y.OD(t,this.b,this.c)),b=Y.HD(new Y.OD(t,this.b,this.a)),c=Y.HD(new Y.OD(t,this.b,this.c)),P=b.a*c.a+b.b*c.b+b.c*c.c,d=1-P*P,g=j.Math.max(d>0?j.Math.sqrt(d):0,gA),A=sn*j.Math.acos(P)-this.j,h=sn*(this.d*(L-this.g)+this.e*(s-this.i)),r=1/L*(c.a-P*b.a),i=1/L*(c.b-P*b.b),D=1/L*(c.c-P*b.c),u=1/s*(b.a-P*c.a),o=1/s*(b.b-P*c.b),Q=1/s*(b.c-P*c.c),e[3*this.a]+=De*(b.a*this.d*A+r/-g*h),e[3*this.a+1]+=De*(b.b*this.d*A+i/-g*h),e[3*this.a+2]+=De*(b.c*this.d*A+D/-g*h),e[3*this.b]+=De*((-b.a*this.d-c.a*this.e)*A+(-r-u)/-g*h),e[3*this.b+1]+=De*((-b.b*this.d-c.b*this.e)*A+(-i-o)/-g*h),e[3*this.b+2]+=De*((-b.c*this.d-c.c*this.e)*A+(-D-Q)/-g*h),e[3*this.c]+=De*(c.a*this.e*A+u/-g*h),e[3*this.c+1]+=De*(c.b*this.e*A+o/-g*h),e[3*this.c+2]+=De*(c.c*this.e*A+Q/-g*h);},y.a=0,y.b=0,y.c=0,y.d=0,y.e=0,y.g=0,y.i=0,y.j=0,l.Q8=X(355),Y.rD=function(t,e){this.a=new At.WD(this),this.e=new At.vE(),this.b=new At.fE(),this.c=new At.kE(),this.d=new At.pE(this),this.f=new At.CE(),this.i=new At.IE(),this.g=new At.FE(),this.j=new At.ME(),this.k=new At.QE(this,t),this.n=new At.VE(this),this.o=new At.ZE(this,e),this.p=new At.kF();},x(154,1,{154:1},Y.rD),l.R8=X(154),Y.sD=function(t,e,A,P,r,i){var D;this.a=A,this.b=P,this.c=r,this.d=i,D=At.YE(t.o,e,A,P,r,i),this.e=D.a,this.f=D.b,this.g=D.c;},Y.tD=function(t,e){var A,P,r,i,D,u,o,Q,L;for(L=new U.Lj(),A=0;A<e.q;A++)for(D=0;D<e.e[A];D++)for(P=e.i[A][D],u=0;u<e.e[P];u++)if(r=e.i[P][u],A!=r)for(o=0;o<e.e[r];o++)i=e.i[r][o],!(P==i||A==i)&&i>A&&(Q=new Y.sD(t,e,A,P,r,i),(j.Math.abs(Q.e)>.001||j.Math.abs(Q.f)>.001||j.Math.abs(Q.g)>.001)&&w.nzb(L.a,Q));return L;},x(356,1,Un,Y.sD),y.bc=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b;return o=new Y.OD(t,this.a,this.b),Q=new Y.OD(t,this.c,this.b),L=new Y.OD(t,this.b,this.c),s=new Y.OD(t,this.d,this.c),h=new Y.JD(o.b*Q.c-o.c*Q.b,o.c*Q.a-o.a*Q.c,o.a*Q.b-o.b*Q.a),b=new Y.JD(L.b*s.c-L.c*s.b,L.c*s.a-L.a*s.c,L.a*s.b-L.b*s.a),D=(h.a*b.a+h.b*b.b+h.c*b.c)/(j.Math.sqrt(h.a*h.a+h.b*h.b+h.c*h.c)*j.Math.sqrt(b.a*b.a+b.b*b.b+b.c*b.c)),r=2*D*D-1,i=D*(2*r-1),u=.5*(this.e*(1+D)+this.f*(1-r)+this.g*(1+i)),u;},y.cc=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;D=F(k(l.U8,1),C1,22,0,[new Y.OD(t,this.b,this.a),new Y.OD(t,this.b,this.c),new Y.OD(t,this.c,this.b),new Y.OD(t,this.c,this.d)]),h=F(k(l.U8,1),C1,22,0,[Y.DD(D[0],D[1]),Y.DD(D[2],D[3])]),P=F(k(l.b7,1),jt,6,15,[Y.GD(h[0]),Y.GD(h[1])]),!(j.Math.abs(P[0])<zA||j.Math.abs(P[1])<zA)&&(h[0]=Y.HD(h[0]),h[1]=Y.HD(h[1]),A=Y.FD(h[0],h[1]),L=1-A*A,Q=L>0?j.Math.sqrt(L):0,u=2*Q*A,o=3*Q-4*Q*L,i=.5*(-this.e*Q+2*this.f*u-3*this.g*o),s=-i*(j.Math.abs(Q)<zA?1/A:1/Q),r=F(k(l.b7,1),jt,6,15,[1/P[0]*(h[1].a-A*h[0].a),1/P[0]*(h[1].b-A*h[0].b),1/P[0]*(h[1].c-A*h[0].c),1/P[1]*(h[0].a-A*h[1].a),1/P[1]*(h[0].b-A*h[1].b),1/P[1]*(h[0].c-A*h[1].c)]),e[3*this.a]+=s*(r[2]*D[1].b-r[1]*D[1].c),e[3*this.a+1]+=s*(r[0]*D[1].c-r[2]*D[1].a),e[3*this.a+2]+=s*(r[1]*D[1].a-r[0]*D[1].b),e[3*this.b]+=s*(r[1]*(D[1].c-D[0].c)+r[2]*(D[0].b-D[1].b)+r[4]*-D[3].c+r[5]*D[3].b),e[3*this.b+1]+=s*(r[0]*(D[0].c-D[1].c)+r[2]*(D[1].a-D[0].a)+r[3]*D[3].c+r[5]*-D[3].a),e[3*this.b+2]+=s*(r[0]*(D[1].b-D[0].b)+r[1]*(D[0].a-D[1].a)+r[3]*-D[3].b+r[4]*D[3].a),e[3*this.c]+=s*(r[1]*D[0].c+r[2]*-D[0].b+r[4]*(D[3].c-D[2].c)+r[5]*(D[2].b-D[3].b)),e[3*this.c+1]+=s*(r[0]*-D[0].c+r[2]*D[0].a+r[3]*(D[2].c-D[3].c)+r[5]*(D[3].a-D[2].a)),e[3*this.c+2]+=s*(r[0]*D[0].b+r[1]*-D[0].a+r[3]*(D[3].b-D[2].b)+r[4]*(D[2].a-D[3].a)),e[3*this.d]+=s*(r[4]*D[2].c-r[5]*D[2].b),e[3*this.d+1]+=s*(r[5]*D[2].a-r[3]*D[2].c),e[3*this.d+2]+=s*(r[3]*D[2].b-r[4]*D[2].a));},y.a=0,y.b=0,y.c=0,y.d=0,y.e=0,y.f=0,y.g=0,l.S8=X(356),Y.wD=function(t,e){var A,P,r,i,D;return i=At.jF(e.p,t.b),D=At.jF(e.p,t.d),A=At.fF(e.p,t.b),P=At.fF(e.p,t.d),r=(i-D)/(i+D),.5*(i+D)*(1+(A==68||P==68?0:.2*(1-j.Math.exp(-12*r*r))));},Y.xD=function(t,e,A){var P,r,i,D,u,o,Q;return i=At.gF(e.p,t.b),D=At.gF(e.p,t.d),P=At.eF(e.p,t.b),r=At.eF(e.p,t.d),u=At.iF(e.p,t.b),o=At.iF(e.p,t.d),Q=A*A,181.16*i*D*P*r/((j.Math.sqrt(P/u)+j.Math.sqrt(r/o))*Q*Q*Q);},Y.yD=function(t,e,A,P){var r,i;this.g=e,this.b=e.a[A],this.d=e.a[P],this.a=A,this.c=P,r=Y.wD(this,t),i=Y.xD(this,t,r),this.e=At.fF(t.p,this.b),this.f=At.fF(t.p,this.d),(this.e==68&&this.f==65||this.e==65&&this.f==68)&&(r=r*.8,i*=.5),this.i=r,this.j=i;},Y.zD=function(t,e,A,P){var r,i,D,u,o;for(o=new U.Lj(),i=0;i<e.q;i++)for(D=i+1;D<e.q;D++)r=new Y.hD(i,D),u=U.snb(A.a,r)!=null?U.snb(A.a,r):(Y.dD(),Y.cD),(u==(Y.dD(),Y.$C)||u==Y.cD)&&Y.GD(new Y.LD(e,i,D))<P&&U.rj(o,new Y.yD(t,e,i,D));return o;},x(357,1,Un,Y.yD),y.bc=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b;return P&&(n.Sl(this.g,this.a)==1||n.Sl(this.g,this.c)==1)?0:(o=Y.GD(new Y.OD(t,this.a,this.c)),Q=o*o,L=Q*Q*Q*o,r=1.07*this.i/(o+si*this.i),i=r*r,D=i*i*i*r,h=this.i*this.i,b=h*h*h*this.i,u=1.12*b/(L+hi*b)-2,s=D*u*this.j,s);},y.cc=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b;for(r=Y.GD(new Y.OD(t,this.a,this.c)),D=r/this.i,u=D*D,o=u*u*u,Q=o*D,L=Q+hi,s=1.07/(D+1.07-1),h=s*s,b=h*h*h*s,A=this.j/this.i*b*(Ti*o/(L*L)+(Ti/L+14)/(D+si)),i=0;i<3;i++)P=.01*this.i,r>0&&(P=A*(t[3*this.a+i]-t[3*this.c+i])/r),e[3*this.a+i]+=P,e[3*this.c+i]-=P;},y.a=0,y.b=0,y.c=0,y.d=0,y.e=0,y.f=0,y.i=0,y.j=0,l.T8=X(357),Y.CD=function(t,e){return j.Math.acos((t.a*e.a+t.b*e.b+t.c*e.c)/(j.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)*j.Math.sqrt(e.a*e.a+e.b*e.b+e.c*e.c)));},Y.DD=function(t,e){return new Y.JD(t.b*e.c-t.c*e.b,t.c*e.a-t.a*e.c,t.a*e.b-t.b*e.a);},Y.ED=function(t,e){return j.Math.sqrt((t.a-e.a)*(t.a-e.a)+(t.b-e.b)*(t.b-e.b)+(t.c-e.c)*(t.c-e.c));},Y.FD=function(t,e){return t.a*e.a+t.b*e.b+t.c*e.c;},Y.GD=function(t){return j.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c);},Y.HD=function(t){return j.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)>0?new Y.JD(t.a/j.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),t.b/j.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),t.c/j.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)):new Y.JD(0,0,0);},Y.ID=function(t,e){return new Y.JD(e.a-t.a,e.b-t.b,e.c-t.c);},Y.JD=function(t,e,A){this.a=t,this.b=e,this.c=A;},Y.KD=function(t,e){this.a=t.J[e].a,this.b=t.J[e].b,this.c=t.J[e].c;},Y.LD=function(t,e,A){Y.MD.call(this,Y.ID(new Y.KD(t,e),new Y.KD(t,A)));},Y.MD=function(t){this.a=t.a,this.b=t.b,this.c=t.c;},Y.ND=function(t,e){this.a=t[3*e],this.b=t[3*e+1],this.c=t[3*e+2];},Y.OD=function(t,e,A){Y.MD.call(this,Y.ID(new Y.ND(t,e),new Y.ND(t,A)));},x(22,1,{22:1},Y.JD,Y.KD,Y.LD,Y.ND,Y.OD),y.lb=function(t){var e;return t===this?true:Jt(t,22)?(e=t,j.Math.abs(this.a-e.a)<.01&&j.Math.abs(this.b-e.b)<.01&&j.Math.abs(this.c-e.c)<.01):false;},y.nb=function(){return Ft(this.a)+Ft(this.b)+Ft(this.c);},y.a=0,y.b=0,y.c=0,l.U8=X(22),At.RD=function(t,e,A,P,r,i){var D,u,o,Q,L,s,h,b,c;for(c=F(k(l.b7,1),jt,6,15,[0,0,0]),Q=F(k(l.b7,1),jt,6,15,[0,0,0]),u=F(k(l.d7,1),H,6,15,[e.C[A],e.C[P],e.C[r]]),o=1.75,L=0;L<3;L++)switch(u[L]){case 1:c[L]=1.395;break;case 6:c[L]=2.494,Q[L]=1.016;break;case 7:c[L]=2.711,Q[L]=1.113;break;case 8:c[L]=3.045,Q[L]=1.337;break;case 9:c[L]=2.847;break;case 14:c[L]=2.35,Q[L]=.811;break;case 15:c[L]=2.35,Q[L]=1.068;break;case 16:c[L]=2.98,Q[L]=1.249;break;case 17:c[L]=2.909,Q[L]=1.078;break;case 35:c[L]=3.017;break;case 53:c[L]=3.086;}return s=At.oE(t.a.d,e,A,P),h=At.oE(t.a.d,e,P,r),D=(s-h)*(s-h)/((s+h)*(s+h)),b=Ai*i,i1.pF(e,A,P,r,4)?o*=.85:i1.pF(e,A,P,r,3)&&(o*=.05),o*c[0]*Q[1]*c[2]/((s+h)*b*b*j.Math.exp(2*D));},At.SD=function(t,e,A,P,r){var i;if(i1.pF(e,A,P,r,3))return 60;if(i1.pF(e,A,P,r,4))return 90;switch(i=e.a[P],At.$D(t.a.b,i)){case 2:if(e.C[P]==8)return 105;if(At.aE(t.a.b,i))return 180;break;case 3:if(At.eE(t.a.b,i)==3&&At.bE(t.a.b,i)==0)return e.C[P]==7?107:92;break;case 4:return 109.45;}return 120;},At.TD=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b;for(i=e.a[A],u=e.a[P],o=e.a[r],L=i1.oF(t.a,e,A,P,r),h=-1,s=0;s<5&&h<0;s++)D=t.a.g.a[i-1][s],Q=t.a.g.a[o-1][s],D>Q&&(D=(b=T.uib(Q),T.uib(Q=D),b).a),h=Y.WC(F(k(l.d7,1),H,6,15,[2,1,3,0]),F(k(l.d7,1),H,6,15,[u,D,Q,L]),t);return h;},At.UD=function(t,e,A,P,r){var i;return i=At.TD(t,e,A,P,r),i<0?At.RD(t,e,A,P,r,At.SD(t,e,A,P,r)):j.Math.abs(T.Rhb(t.b[i][4]))<.001?At.RD(t,e,A,P,r,T.Rhb(t.b[i][5])):T.Rhb(t.b[i][4]);},At.VD=function(t,e,A,P,r){var i;return i=At.TD(t,e,A,P,r),i<0?At.SD(t,e,A,P,r):T.Rhb(t.b[i][5]);},At.WD=function(t){this.b=Y.mC("/resources/forcefield/mmff94/angle.csv"),this.a=t;},x(377,1,{},At.WD),y.jc=function(t,e){return T.Shb(this.b[t][e]);},y.kc=function(){return this.b.length;},l.V8=X(377),At.ZD=function(t,e){return t.a[Y.UC(0,e,0,t.a.length,true,t)][6]>0;},At.$D=function(t,e){return t.a[Y.UC(0,e,0,t.a.length,true,t)][2];},At._D=function(t){return t.a.length;},At.aE=function(t,e){var A;return A=Y.UC(0,e,0,t.a.length,true,t),A>=0&&t.a[A][7]>0;},At.bE=function(t,e){return t.a[Y.UC(0,e,0,t.a.length,true,t)][5];},At.cE=function(t,e){return t.a[Y.UC(0,e,0,t.a.length,true,t)][4];},At.dE=function(t,e){return t.a[Y.UC(0,e,0,t.a.length,true,t)][8]>0;},At.eE=function(t,e){return t.a[Y.UC(0,e,0,t.a.length,true,t)][3];},At.fE=function(){this.a=Y.nC("/resources/forcefield/mmff94/atom.csv");},x(372,1,{},At.fE),y.jc=function(t,e){return this.a[t][e];},y.kc=function(){return At._D(this);},l.W8=X(372),At.iE=function(t,e){return T.Rhb(t.a[e][3]);},At.jE=function(t,e){return T.Rhb(t.a[e][2]);},At.kE=function(){this.a=Y.mC("/resources/forcefield/mmff94/bndk.csv");},x(384,1,{},At.kE),y.jc=function(t,e){return T.Shb(this.a[t][e]);},y.kc=function(){return this.a.length;},l.X8=X(384),At.nE=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g;return L=i1.zF(t.a,e,n.Wo(e,A,P)),i=e.a[A],u=e.a[P],i>u&&(i=(c=T.uib(u),T.uib(u=i),c).a),h=Y.WC(F(k(l.d7,1),H,6,15,[1,2,0]),F(k(l.d7,1),H,6,15,[i,u,L]),t),h>=0?T.Rhb(t.b[h][3]):(r=e.C[A],D=e.C[P],r>D&&(r=(b=T.uib(D),T.uib(D=r),b).a),g=At.oE(t,e,A,P),Q=Y.WC(F(k(l.d7,1),H,6,15,[0,1]),F(k(l.d7,1),H,6,15,[r,D]),t.a.c),Q>=0?(s=j.Math.pow(At.jE(t.a.c,Q)/g,6),At.iE(t.a.c,Q)*s):(o=Y.WC(F(k(l.d7,1),H,6,15,[0,1]),F(k(l.d7,1),H,6,15,[Y.HC(r),Y.HC(D)]),t.a.j),j.Math.pow(10,-(g-At.LE(t.a.j,o,2))/At.LE(t.a.j,o,3))));},At.oE=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;return L=i1.zF(t.a,e,n.Wo(e,A,P)),D=e.a[A],Q=e.a[P],D>Q&&(D=(d=T.uib(Q),T.uib(Q=D),d).a),c=Y.WC(F(k(l.d7,1),H,6,15,[1,2,0]),F(k(l.d7,1),H,6,15,[D,Q,L]),t),c>=0?T.Rhb(t.b[c][4]):(i=e.C[A],o=e.C[P],i>o&&(i=(g=T.uib(o),T.uib(o=i),g).a),r=Y.VC(i,t.a.f),u=Y.VC(o,t.a.f),m=At.BE(t.a.f,r),C=At.BE(t.a.f,u),h=At.zE(t.a.f,r),b=At.zE(t.a.f,u),s=i==1||o==1?.05:.085,m+C-s*j.Math.pow(j.Math.abs(h-b),1.4));},At.pE=function(t){this.b=Y.mC("/resources/forcefield/mmff94/bond.csv"),this.a=t;},x(380,1,{},At.pE),y.jc=function(t,e){return T.Shb(this.b[t][e]);},y.kc=function(){return this.b.length;},l.Y8=X(380),At.sE=function(t,e,A,P){return Y.UC(1,e,0,A,P,new At.wE(t));},At.tE=function(t,e){return T.Rhb(t.b[e-1][2]);},At.uE=function(t,e,A,P){var r,i,D,u,o,Q,L;return L=A>P?1:-1,Q=A>P?P:A,o=A>P?A:P,i=At.sE(t,Q,t.a.length,true),r=At.sE(t,Q,t.a.length,false),i==-1||r==-1||(u=Y.UC(2,o,i,r+1,true,new At.wE(t)),D=Y.UC(2,o,i,r+1,false,new At.wE(t)),u==-1||D==-1)?T.Rhb(t.b[A-1][1])-T.Rhb(t.b[P-1][1]):e==0&&T.Shb(t.a[u][0])==0?L*T.Rhb(t.a[u][3]):e==1&&T.Shb(t.a[D][0])==1?L*T.Rhb(t.a[D][3]):T.Rhb(t.b[A-1][1])-T.Rhb(t.b[P-1][1]);},At.vE=function(){this.b=Y.mC("/resources/forcefield/mmff94/pbci.csv"),this.a=Y.mC("/resources/forcefield/mmff94/bci.csv");},x(378,1,{},At.vE),l.$8=X(378),At.wE=function(t){this.a=t;},x(196,1,{},At.wE),y.jc=function(t,e){return T.Shb(this.a.a[t][e]);},y.kc=function(){return this.a.a.length;},l.Z8=X(196),At.zE=function(t,e){return T.Rhb(t.a[e][2]);},At.AE=function(t){return t.a.length;},At.BE=function(t,e){return T.Rhb(t.a[e][1]);},At.CE=function(){this.a=Y.mC("/resources/forcefield/mmff94/covrad.csv");},x(385,1,{},At.CE),y.jc=function(t,e){return T.Shb(this.a[t][e]);},y.kc=function(){return At.AE(this);},l._8=X(385),At.FE=function(){this.a=Y.nC("/resources/forcefield/mmff94/def.csv");},x(387,1,{},At.FE),l.a9=X(387),At.GE=function(t,e,A,P,r){var i,D,u,o;return i=Y.GC(e.C[A]),D=Y.GC(e.C[P]),u=Y.GC(e.C[r]),i>u&&(i=(o=T.uib(u),T.uib(u=i),o).a),Y.WC(F(k(l.d7,1),H,6,15,[1,0,2]),F(k(l.d7,1),H,6,15,[D,i,u]),t);},At.HE=function(t,e,A,P,r){var i,D,u,o;return i=Y.GC(e.C[A]),D=Y.GC(e.C[r]),u=i>D?4:3,o=At.GE(t,e,A,P,r),o>=0?T.Rhb(t.a[o][u]):0;},At.IE=function(){this.a=Y.mC("/resources/forcefield/mmff94/dfsb.csv");},x(386,1,{},At.IE),y.jc=function(t,e){return T.Shb(this.a[t][e]);},y.kc=function(){return this.a.length;},l.b9=X(386),At.LE=function(t,e,A){return T.Rhb(t.a[e][A]);},At.ME=function(){this.a=Y.mC("/resources/forcefield/mmff94/herschbachlaurie.csv");},x(388,1,{},At.ME),y.jc=function(t,e){return T.Shb(this.a[t][e]);},y.kc=function(){return this.a.length;},l.c9=X(388),At.PE=function(t,e,A,P,r,i){var D,u,o,Q,L,s;for(D=e.a[A],L=F(k(l.d7,1),H,6,15,[e.a[P],e.a[r],e.a[i]]),u=0;u<4;u++){for(s=z(l.d7,H,6,3,15,1),Q=0;Q<3;Q++)s[Q]=t.a.g.a[L[Q]-1][u+1];if(w.qzb(s,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),o=Y.WC(F(k(l.d7,1),H,6,15,[1,0,2,3]),F(k(l.d7,1),H,6,15,[D,s[0],s[1],s[2]]),t),o>=0)return T.Rhb(t.b[o][4]);}return 0;},At.QE=function(t,e){this.b=Y.mC(e),this.a=t;},x(379,1,{},At.QE),y.jc=function(t,e){return T.Shb(this.b[t][e]);},y.kc=function(){return this.b.length;},l.d9=X(379),At.TE=function(t,e,A,P,r){var i,D,u,o,Q;return i=e.a[A],D=e.a[P],u=e.a[r],o=i1.nF(t.a,e,A,P,r),i>u&&(i=(Q=T.uib(u),T.uib(u=i),Q).a),Y.WC(F(k(l.d7,1),H,6,15,[2,1,3,0]),F(k(l.d7,1),H,6,15,[D,i,u,o]),t);},At.UE=function(t,e,A,P,r){var i,D,u,o,Q,L;return i=e.a[A],D=e.a[r],o=i1.AF(t.a,e,A,P),Q=i1.AF(t.a,e,P,r),L=At.TE(t,e,A,P,r),u=i>D||i==D&&o<Q?1:0,L>=0?T.Rhb(t.b[L][4+u]):At.HE(t.a.i,e,A,P,r);},At.VE=function(t){this.b=Y.mC("/resources/forcefield/mmff94/stbn.csv"),this.a=t;},x(381,1,{},At.VE),y.jc=function(t,e){return T.Shb(this.b[t][e]);},y.kc=function(){return this.b.length;},l.e9=X(381),At.YE=function(t,e,A,P,r,i){var D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt;for(L=e.a[A],s=e.a[P],h=e.a[r],b=e.a[i],ot=i1.DF(t.a,e,A,P,r,i),rt=ot>10?ot/10|0:ot,yt=ot>10?ot-rt*10:0,K=-1,N=0,O=0,I=0,S=rt,Dt=5;(N<Dt&&(K==-1||Dt==4)||N==4&&rt==5&&yt>0)&&(Dt==5&&N==4&&(Dt=4,N=0,S=yt),N==1?(O=1,I=3):N==2?(O=3,I=1):(O=N,I=N),m=t.a.g.a[L-1][j.Math.min(O+1,4)],C=s,v=h,M=t.a.g.a[b-1][j.Math.min(I+1,4)],C>v?(v=(Qt=T.uib(C),T.uib(C=v),Qt).a,M=(ut=T.uib(m),T.uib(m=M),ut).a):C==v&&m>M&&(M=(ut=T.uib(m),T.uib(m=M),ut).a),K=Y.WC(F(k(l.d7,1),H,6,15,[2,3,1,4,0]),F(k(l.d7,1),H,6,15,[C,v,m,M,S]),t),!(K!=-1&&Dt==4));)++N;if(K>=0)return new At.cF(t,K);for(d=n.Wo(e,P,r),u=F(k(l.b7,1),jt,6,15,[0,0]),o=F(k(l.b7,1),jt,6,15,[0,0]),Q=F(k(l.b7,1),jt,6,15,[0,0]),c=F(k(l.d7,1),H,6,15,[e.C[P],e.C[r]]),D=(At.$D(t.a.b,s)-1)*(At.$D(t.a.b,h)-1),V=0;V<2;V++)switch(c[V]){case 6:u[V]=2,o[V]=2.12;break;case 7:u[V]=2,o[V]=1.5;break;case 8:u[V]=2,o[V]=.2,Q[V]=2;break;case 14:u[V]=1.25,o[V]=1.22;break;case 15:u[V]=1.25,o[V]=2.4;break;case 16:u[V]=1.25,o[V]=.49,Q[V]=8;}return At.aE(t.a.b,s)||At.aE(t.a.b,h)?new At.bF(0,0):At.ZD(t.a.b,s)&&At.ZD(t.a.b,h)&&d<e.g&&n.ot(e.p,d)?(g=At.eE(t.a.b,s)==3&&At.eE(t.a.b,h)==4||At.eE(t.a.b,s)==4&&At.eE(t.a.b,h)==3?3:6,Lt=At.cE(t.a.b,s)==0&&At.cE(t.a.b,h)==0?.5:.3,new At.bF(g*Lt*j.Math.sqrt(u[0]*u[1]),0)):n.em(e,d)==2?(Lt=At.bE(t.a.b,s)==2&&At.bE(t.a.b,h)==2?1:.4,new At.bF(6*Lt*j.Math.sqrt(u[0]*u[1]),0)):At.$D(t.a.b,s)==4&&At.$D(t.a.b,h)==4?new At.bF(0,j.Math.sqrt(o[0]*o[1])/D):At.$D(t.a.b,s)==4&&At.$D(t.a.b,h)!=4?At.$D(t.a.b,h)==3&&(At.eE(t.a.b,h)==4||At.eE(t.a.b,h)==34||At.bE(t.a.b,h)>0)||At.$D(t.a.b,h)==2&&(At.eE(t.a.b,h)==3||At.bE(t.a.b,h)>0)?new At.aF():new At.bF(0,j.Math.sqrt(o[0]*o[1])/D):At.$D(t.a.b,h)==4&&At.$D(t.a.b,s)!=4?At.$D(t.a.b,s)==3&&(At.eE(t.a.b,s)==4||At.eE(t.a.b,s)==34||At.bE(t.a.b,s)>0)||At.$D(t.a.b,s)==2&&(At.eE(t.a.b,s)==3||At.bE(t.a.b,s)>0)?new At.aF():new At.bF(0,j.Math.sqrt(o[0]*o[1])/D):n.em(e,d)==1&&At.bE(t.a.b,s)>0&&At.bE(t.a.b,h)>0||At.bE(t.a.b,s)>0&&At.cE(t.a.b,h)>0||At.cE(t.a.b,s)>0&&At.bE(t.a.b,h)>0?At.cE(t.a.b,s)>0&&At.cE(t.a.b,h)>0?new At.aF():At.cE(t.a.b,s)>0&&At.bE(t.a.b,h)>0?(Lt=0,At.bE(t.a.b,s)==1?Lt=.5:Y.GC(c[0])==2&&Y.GC(c[1])==2?Lt=.3:(Y.GC(c[0])!=2||Y.GC(c[1])!=2)&&(Lt=.15),new At.bF(6*Lt*j.Math.sqrt(u[0]*u[1]),0)):At.cE(t.a.b,h)>0&&At.bE(t.a.b,s)>0?(Lt=0,At.bE(t.a.b,h)==1?Lt=.5:Y.GC(c[0])==2&&Y.GC(c[1])==2?Lt=.3:(Y.GC(c[0])!=2||Y.GC(c[1])!=2)&&(Lt=.15),new At.bF(6*Lt*j.Math.sqrt(u[0]*u[1]),0)):(At.bE(t.a.b,s)==1||At.bE(t.a.b,h)==1)&&(c[0]!=6||c[1]!=6)?new At.bF(2.4000000000000004*j.Math.sqrt(u[0]*u[1]),0):new At.bF(.8999999999999999*j.Math.sqrt(u[0]*u[1]),0):(c[0]==8||c[0]==16)&&(c[1]==8||c[1]==16)?new At.bF(-j.Math.sqrt(Q[0]*Q[1]),0):new At.bF(0,j.Math.sqrt(o[0]*o[1])/D);},At.ZE=function(t,e){this.b=Y.mC(e),this.a=t;},x(382,1,{},At.ZE),y.jc=function(t,e){return T.Shb(this.b[t][e]);},y.kc=function(){return this.b.length;},l.g9=X(382),At.aF=function(){this.a=0,this.b=0,this.c=0;},At.bF=function(t,e){this.a=0,this.b=t,this.c=e;},At.cF=function(t,e){this.a=T.Rhb(t.b[e][5]),this.b=T.Rhb(t.b[e][6]),this.c=T.Rhb(t.b[e][7]);},x(49,1,{},At.aF,At.bF,At.cF),y.ob=function(){return this.a+","+this.b+","+this.c;},y.a=0,y.b=0,y.c=0,l.f9=X(49),At.eF=function(t,e){return T.Rhb(t.a[Y.UC(0,e,0,t.a.length,true,t)][1]);},At.fF=function(t,e){return t.a[Y.UC(0,e,0,t.a.length,true,t)][5].a;},At.gF=function(t,e){return T.Rhb(t.a[Y.UC(0,e,0,t.a.length,true,t)][4]);},At.hF=function(t){return t.a.length;},At.iF=function(t,e){return T.Rhb(t.a[Y.UC(0,e,0,t.a.length,true,t)][2]);},At.jF=function(t,e){var A;return A=Y.UC(0,e,0,t.a.length,true,t),T.Rhb(t.a[A][3])*j.Math.pow(T.Rhb(t.a[A][1]),.25);},At.kF=function(){this.a=Y.mC("/resources/forcefield/mmff94/vanderwaals.csv");},x(383,1,{},At.kF),y.jc=function(t,e){return T.Shb(this.a[t][e]);},y.kc=function(){return At.hF(this);},l.h9=X(383),i1.nF=function(t,e,A,P,r){var i,D,u,o,Q,L,s;switch(i=e.a[A],D=e.a[r],Q=i1.AF(t,e,A,P),s=i1.AF(t,e,P,r),o=i<=D?Q:s,L=i<D?s:Q,u=i1.oF(t,e,A,P,r),u){case 1:return o>0||o==L?1:2;case 2:return 3;case 3:return 5;case 4:return 4;case 5:return o>0||o==L?6:7;case 6:return 8;case 7:return o>0||o==L?9:10;case 8:return 11;}return 0;},i1.oF=function(t,e,A,P,r){var i,D;return i=i1.AF(t,e,A,P)+i1.AF(t,e,P,r),D=i,i1.pF(e,A,P,r,3)?D+=i>0?4:3:i1.pF(e,A,P,r,4)&&(D+=i>0?6:4),D;},i1.pF=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h;if(h=(n.aw(t,7),t.p),Q=new U.wtb(),U.vtb(Q,T.uib(e)),U.vtb(Q,T.uib(A)),U.vtb(Q,T.uib(P)),n.Wo(t,e,A)>=0&&n.Wo(t,A,P)>=0){for(L=0;L<h.j.a.length;L++)if(U.wj(h.k,L).length==r){for(s=new U.wtb(),D=U.wj(h.j,L),u=0,o=D.length;u<o;++u)i=D[u],U.vtb(s,T.uib(i));if(U.Ii(s,Q))return  true;}}return  false;},i1.qF=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt,mt,y1,u1,Mt,Zt,m1,Et,f1,x1,s1,Q1,I1,G1,v0,K1,L1,D0,T0,j0,pe,ge,ne,Ae,R0,Re,K0,C0,Je,jn,zt,Wt,ze,Bn,mn,kn;if(e<t.f&&n.nt(t.p,e)){if(i1.xF(t,e,5)){if(g=new U.Mj(t.q),C=new U.Mj(t.q),c=false,I=false,ut=false,t.C[e]==6||t.C[e]==7){for(O=0;O<t.e[e];O++)if(zt=t.i[e][O],!!i1.xF(t,zt,5))for(i1.vF(t,e,zt)&&(t.C[zt]==8||t.C[zt]==16||t.C[zt]==7&&t.e[zt]+n.np(t,zt)==3&&!i1.wF(t,zt))&&U.rj(g,T.uib(zt)),I1=0;I1<t.e[zt];I1++)Wt=t.i[zt][I1],Wt!=e&&i1.xF(t,Wt,5)&&i1.vF(t,e,Wt)&&(t.C[Wt]==8||t.C[Wt]==16||t.C[Wt]==7&&t.e[Wt]+n.np(t,Wt)==3&&!i1.wF(t,Wt))&&U.rj(C,T.uib(Wt));for(b=new U.wqb(g);b.a<b.c.a.length;)if(h=U.vqb(b).a,t.C[h]==8||t.C[h]==16){I=true;break;}for(m=new U.wqb(C);m.a<m.c.a.length;)if(d=U.vqb(m).a,t.C[d]==8||t.C[d]==16){ut=true;break;}for(V=0;V<g.a.length;V++)for(Q1=V;Q1<C.a.length;Q1++)if(i1.vF(t,(w.Kzb(V,g.a.length),g.a[V]).a,(w.Kzb(Q1,C.a.length),C.a[Q1]).a)){c=true;break;}}switch(t.C[e]){case 6:if(C.a.length==0){for(T0=0,v0=0,L1=0,D0=0,V=0;V<t.e[e];V++)zt=t.i[e][V],t.C[zt]==7&&t.e[zt]+n.np(t,zt)==3&&(++T0,t.s[zt]>0&&!i1.wF(t,zt)&&++v0,i1.xF(t,zt,5)&&++L1,i1.xF(t,zt,6)&&++D0);if((T0==2&&L1>0||T0==3&&L1==2)&&v0>0&&D0==0)return 80;}if(!(g.a.length==0^C.a.length==0)){for(kn=true,mn=true,V=0;V<t.e[e];V++)zt=t.i[e][V],(t.C[zt]!=6||!i1.tF(t,zt))&&(kn=false),i1.vF(t,e,zt)&&!(zt<t.f&&n.nt(t.p,zt))&&(mn=false);if(g.a.length==0&&C.a.length==0&&!kn&&mn||g.a.length!=0&&g.a.length!=0&&(!c||!I&&!ut))return 78;}if(g.a.length!=0&&(C.a.length==0||I))return 63;if(C.a.length!=0&&(g.a.length==0||ut))return 64;break;case 7:if(i1.wF(t,e))return 82;if(g.a.length==0&&C.a.length==0)return t.e[e]+n.np(t,e)==3?39:76;if(t.e[e]+n.np(t,e)==3&&g.a.length==0^C.a.length==0)return 81;if(g.a.length!=0&&(C.a.length==0||I))return 65;if(C.a.length!=0&&(g.a.length==0||ut))return 66;if(g.a.length!=0&&C.a.length!=0)return 79;break;case 8:return 59;case 16:return 44;}}if(i1.xF(t,e,6))switch(t.C[e]){case 6:return 37;case 7:return i1.wF(t,e)?69:t.e[e]+n.np(t,e)==3?58:38;}}switch(t.C[e]){case 3:if(t.j[e]==0)return 92;break;case 6:if(t.e[e]+n.np(t,e)==4)return (t.p&&e<t.f?n.ct(t.p,e):0)==3?22:(t.p&&e<t.f?n.ct(t.p,e):0)==4?20:1;if(t.e[e]+n.np(t,e)==3){for(j0=0,pe=0,ne=0,K0=0,M=0,O=0;O<t.e[e];O++)zt=t.i[e][O],n.em(t,n.Wo(t,e,zt))==2&&(M=t.C[zt]),t.e[zt]+n.np(t,zt)==1?(t.C[zt]==8&&++ne,t.C[zt]==16&&++K0):t.C[zt]==7&&(t.e[zt]+n.np(t,zt)==3?++pe:t.e[zt]+n.np(t,zt)==2&&n.em(t,n.Wo(t,e,zt))==2&&++j0);return pe>=2&&j0==0&&M==7?57:ne==2||K0==2?41:(t.p&&e<t.f?n.ct(t.p,e):0)==4&&M==6?30:M==7||M==8||M==15||M==16?3:2;}if(t.e[e]+n.np(t,e)==2)return 4;if(t.e[e]+n.np(t,e)==1)return 60;break;case 7:for(Je=0,f1=false,K=0;K<t.e[e];K++)if(zt=t.i[e][K],t.C[zt]==8&&t.e[zt]+n.np(t,zt)==1&&++Je,n.up(t,e)+n.np(t,e)>=3&&(t.C[zt]==15||t.C[zt]==16)){for(Re=0,Q1=0;Q1<t.e[zt];Q1++)Wt=t.i[zt][Q1],t.C[Wt]==8&&t.e[Wt]+n.np(t,Wt)==1&&++Re;f1||(f1=Re>=2);}if(t.e[e]+n.np(t,e)==4)return i1.wF(t,e)?68:34;if(t.e[e]+n.np(t,e)==3){if(n.up(t,e)+n.np(t,e)>=4){for(S=false,V=0;V<t.e[e];V++)if(zt=t.i[e][V],n.em(t,n.Wo(t,e,zt))==2&&(S=t.C[zt]==7||t.C[zt]==6,t.C[zt]==6))for(Q1=0;S&&Q1<t.e[zt];Q1++)Wt=t.i[zt][Q1],Wt!=e&&(S=!(t.C[Wt]==7&&t.e[Wt]+n.np(t,Wt)==3));if(Je==1)return 67;if(Je>=2)return 45;if(S)return 54;}if(n.up(t,e)+n.np(t,e)>=3){for(it=false,bt=false,ct=false,wt=false,Ot=false,xt=false,A=0,P=0,r=0,i=0,u=0,L=0,V=0;V<t.e[e];V++)if(zt=t.i[e][V],t.C[zt]==6){for(Ot=true,zt<t.f&&n.nt(t.p,zt)&&(t.p&&zt<t.f?n.ct(t.p,zt):0)==6&&(xt=true),r=0,i=0,u=0,L=0,v0=0,N=0,Q1=0;Q1<t.e[zt];Q1++){if(Wt=t.i[zt][Q1],v=n.Wo(t,zt,Wt),n.em(t,v)==2&&(t.C[Wt]==8||t.C[Wt]==16)&&(it=true),(n.em(t,v)==2||v<t.g&&n.ot(t.p,v)&&(t.C[Wt]==6||t.C[Wt]==7&&i1.uF(t,Wt)==1))&&(A=t.C[Wt]),n.em(t,v)==3&&(P=t.C[Wt]),t.C[Wt]==7&&t.e[Wt]+n.np(t,Wt)==3){for(t.s[Wt]==1&&++v0,i1.xF(t,zt,6)&&++N,Q=0,G1=0;G1<t.e[Wt];G1++)ze=t.i[Wt][G1],t.C[ze]==8&&++Q;Q<2&&++i;}t.C[Wt]==7&&t.e[Wt]+n.np(t,Wt)==2&&(n.em(t,v)==2||v<t.g&&n.ot(t.p,v))&&++r,Wt<t.f&&n.nt(t.p,Wt)&&(t.C[Wt]==8&&++u,t.C[Wt]==16&&++L);}A==7&&(i==2&&r==0&&v0>0&&N==0&&t.e[zt]+n.np(t,zt)<4&&(bt=true),i==3&&(ct=true));}else if(t.C[zt]==7){for(D=0,o=0,s=0,Q1=0;Q1<t.e[zt];Q1++)if(Wt=t.i[zt][Q1],v=n.Wo(t,zt,Wt),n.em(t,v)==2){if(t.C[Wt]==6){for(G1=0;G1<t.e[Wt];G1++)ze=t.i[Wt][G1],ze!=e&&(t.C[ze]==7?++D:t.C[ze]==8?++o:t.C[ze]==16&&++s);D==0&&o==0&&s==0&&!xt&&(wt=true);}t.C[Wt]==7&&!xt&&(wt=true);}}if(Ot){if(P==7&&(f1=true),bt)return 55;if(ct)return 56;if(!it&&!f1&&(u==0&&L==0&&xt||A==6||A==7||A==15||P==6))return 40;}if(!f1&&(it||wt))return 10;}}if(t.e[e]+n.np(t,e)==2){if(n.up(t,e)+n.np(t,e)==4){for(rt=false,V=0;!rt&&V<t.e[e];V++)zt=t.i[e][V],rt=n.em(t,n.Wo(t,e,zt))==3;return rt?61:53;}if(n.up(t,e)+n.np(t,e)==3){for(pt=false,ot=false,V=0;V<t.e[e];V++)zt=t.i[e][V],n.em(t,n.Wo(t,e,zt))==2&&(pt=t.C[zt]==8&&Je==1,ot=t.C[zt]==6||t.C[zt]==7);if(pt&&!ot)return 46;if(ot)return 9;}if(n.up(t,e)+n.np(t,e)>=2){for(vt=false,V=0;V<t.e[e];V++)if(zt=t.i[e][V],t.C[zt]==16){for(jn=0,Q1=0;Q1<t.e[zt];Q1++)Wt=t.i[zt][Q1],t.C[Wt]==8&&t.e[Wt]+n.np(t,Wt)==1&&++jn;vt=jn==1;}if(vt)return 48;if(!f1)return 62;}}if(f1)return 43;if(t.e[e]+n.np(t,e)==1){for(Ht=false,yt=false,V=0;!Ht&&!yt&&V<t.e[e];V++)if(zt=t.i[e][V],Ht=n.em(t,n.Wo(t,e,zt))==3,t.C[zt]==7&&t.e[zt]+n.np(t,zt)==2)for(Q1=0;!yt&&Q1<t.e[zt];Q1++)Wt=t.i[zt][Q1],yt=t.C[Wt]==7&&t.e[Wt]+n.np(t,Wt)==2||t.C[Wt]==6&&t.e[Wt]+n.np(t,Wt)==3;if(Ht)return 42;if(yt)return 47;}return 8;case 8:if(t.e[e]+n.np(t,e)==3)return 49;if(t.e[e]+n.np(t,e)==2){if(n.up(t,e)+n.np(t,e)==3)return 51;for(K1=0,V=0;V<t.e[e];V++)zt=t.i[e][V],t.C[zt]==1&&++K1;return K1+n.np(t,e)==2?70:6;}if(t.j[e]<=1){for(ge=0,R0=0,C0=0,Mt=t.e[e]-t.j[e]+n.np(t,e)>0,Dt=false,Qt=false,u1=false,mt=false,Zt=false,$t=false,Nt=false,s1=false,Et=false,x1=false,m1=false,V=0;V<t.e[e]&&!(Mt||Dt||Qt||u1||mt||Zt||$t||Nt||s1||Et||x1||m1);V++){if(zt=t.i[e][V],t.C[zt]==6||t.C[zt]==7||t.C[zt]==16)for(Q1=0;Q1<t.e[zt];Q1++)Wt=t.i[zt][Q1],t.C[Wt]==7&&t.e[Wt]+n.np(t,Wt)==2&&++ge,t.C[Wt]==8&&t.e[Wt]+n.np(t,Wt)==1&&++R0,t.C[Wt]==16&&t.e[Wt]+n.np(t,Wt)==1&&++C0;Mt=t.C[zt]==1,t.C[zt]==6&&(Dt=R0==2,Qt=n.em(t,n.Wo(t,e,zt))==2,u1=n.em(t,n.Wo(t,e,zt))==1&&R0==1),t.C[zt]==7&&(mt=n.em(t,n.Wo(t,e,zt))==2,n.em(t,n.Wo(t,e,zt))==1&&R0==1&&(Zt=t.e[zt]+n.np(t,zt)==2||n.up(t,zt)+n.np(t,zt)==3,$t=n.up(t,zt)+n.np(t,zt)==4),Nt=R0>=2),t.C[zt]==16&&(s1=C0==1,Et=n.em(t,n.Wo(t,e,zt))==1||n.em(t,n.Wo(t,e,zt))==2&&R0+ge>1,x1=n.em(t,n.Wo(t,e,zt))==2&&R0+ge==1),m1=t.C[zt]==15||t.C[zt]==17;}if(u1||Zt||Mt)return 35;if(Dt||Nt||$t||s1||Et||m1)return 32;if(Qt||mt||x1)return 7;}break;case 9:if(t.j[e]==1)return 11;if(t.j[e]==0)return 89;break;case 11:if(t.j[e]==0)return 93;break;case 12:if(t.j[e]==0)return 99;break;case 14:return 19;case 15:if(t.e[e]+n.np(t,e)==4)return 25;if(t.e[e]+n.np(t,e)==3)return 26;if(t.e[e]+n.np(t,e)==2)return 75;break;case 16:if(t.e[e]+n.np(t,e)==3||t.e[e]+n.np(t,e)==4){for(Ae=0,C0=0,Lt=false,V=0;V<t.e[e];V++)zt=t.i[e][V],t.C[zt]==6&&n.em(t,n.Wo(t,e,zt))==2&&(Lt=true),(t.j[zt]==1&&t.C[zt]==8||t.e[zt]+n.np(t,zt)==2&&t.C[zt]==7)&&++Ae,t.j[zt]==1&&t.C[zt]==16&&++C0;return t.e[e]+n.np(t,e)==3&&Ae==2&&Lt||t.e[e]+n.np(t,e)==4?18:Ae>0&&C0>0||Ae==2&&!Lt?73:17;}if(t.e[e]+n.np(t,e)==2){for(y1=false,V=0;V<t.e[e];V++)zt=t.i[e][V],t.C[zt]==8&&n.em(t,n.Wo(t,e,zt))==2&&(y1=true);return y1?74:15;}if(t.e[e]+n.np(t,e)==1){for(Bn=0,Lt=false,V=0;V<t.e[e];V++){for(zt=t.i[e][V],Q1=0;Q1<t.e[zt];Q1++)Wt=t.i[zt][Q1],t.C[Wt]==16&&t.e[Wt]+n.np(t,Wt)==1&&++Bn;t.C[zt]==6&&n.em(t,n.Wo(t,e,zt))==2&&(Lt=true);}return Lt&&Bn!=2?16:72;}break;case 17:if(t.j[e]==4){for(ne=0,V=0;V<t.e[e];V++)zt=t.i[e][V],t.C[zt]==8&&++ne;if(ne==4)return 77;}if(t.j[e]==1)return 12;if(t.j[e]==0)return 90;break;case 19:if(t.j[e]==0)return 94;break;case 20:if(t.j[e]==0)return 96;break;case 26:if(t.j[e]==0){if(t.s[e]==2)return 87;if(t.s[e]==3)return 88;}break;case 29:if(t.j[e]==0){if(t.s[e]==1)return 97;if(t.s[e]==2)return 98;}break;case 30:if(t.j[e]==0)return 95;break;case 35:if(t.j[e]==1)return 13;if(t.j[e]==0)return 91;break;case 53:if(t.j[e]==1)return 14;}return 0;},i1.rF=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;for(P=0;P<t.e[e];P++)switch(L=t.i[e][P],t.C[L]){case 6:return 5;case 7:switch(i1.qF(t,L)){case 8:case 39:case 62:case 67:case 68:return 23;case 34:case 54:case 55:case 56:case 58:case 81:return 36;case 9:return 27;default:return 28;}case 8:switch(i1.qF(t,L)){case 49:return 50;case 51:return 52;case 70:return 31;case 6:for(r=false,i=false,D=false,u=false,o=0;o<t.e[L];o++){if(s=t.i[L][o],t.C[s]==6)for(Q=0;Q<t.e[s];Q++)h=t.i[s][Q],h!=L&&(A=n.Wo(t,s,h),(t.C[h]==6||t.C[h]==7)&&(n.em(t,A)==2||A<t.g&&n.ot(t.p,A))&&(r=true),t.C[h]==8&&n.em(t,A)==2&&(i=true));t.C[s]==15&&(D=true),t.C[s]==16&&(u=true);}if(i||D)return 24;if(r)return 29;if(u)return 33;default:return 21;}case 14:return 5;case 16:case 15:return 71;}return 0;},i1.sF=function(t,e){return t.a[e]>-1?t.a[e]:t.C[e]==1?i1.rF(t,e):i1.qF(t,e);},i1.tF=function(t,e){var A,P;if((t.u[e]&V0)!=0){for(P=(n.aw(t,7),t.p),A=0;A<P.j.a.length;A++)if(U.wj(P.k,A).length==6&&n.pt(P,A,e))return  true;}return  false;},i1.uF=function(t,e){var A,P,r;for(r=(n.aw(t,7),t.p),P=0,A=0;A<r.j.a.length;A++)n.bt(r,A,e)>=0&&++P;return P;},i1.vF=function(t,e,A){var P,r;if((t.u[e]&V0)==0||(t.u[A]&V0)==0)return  false;for(r=(n.aw(t,7),t.p),P=0;P<r.j.a.length;P++)if(U.wj(r.k,P).length==5&&n.pt(r,P,e)&&n.pt(r,P,A))return  true;return  false;},i1.wF=function(t,e){var A,P;if(t.C[e]==7&&t.e[e]+n.np(t,e)>=3){for(A=0;A<t.e[e];A++)if(P=t.i[e][A],t.C[P]==8&&t.e[P]+n.np(t,P)==1)return  true;}return  false;},i1.xF=function(t,e,A){var P,r;if(e<t.f&&n.nt(t.p,e)){for(r=(n.aw(t,7),t.p),P=0;P<r.j.a.length;P++)if(!(U.wj(r.k,P).length!=A||!n.pt(r,P,e))&&t.b[P]==0)return  true;}return  false;},i1.yF=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;if(m=(n.aw(t,7),t.p),!m.e[e])return 1;if(U.wj(m.k,e).length==6){for(P=U.wj(m.j,e),r=0,i=P.length;r<i;++r)if(A=P[r],n.up(t,A)+n.np(t,A)!=t.e[A]+n.np(t,A)+1)return 1;for(o=U.wj(m.k,e),Q=0,L=o.length;Q<L;++Q)if(u=o[Q],s=F(k(l.d7,1),H,6,15,[-1,-1]),n.em(t,u)==1){for(b=0;b<=1;b++)for(D=t.D[b][u],h=0;h<t.e[D];h++)if(c=t.i[D][h],!n.pt(m,e,c)&&n.em(t,n.Wo(t,D,c))==2){s[b]=c;break;}if(s[0]>-1&&s[1]>-1)for(d=0;d<m.j.a.length;d++){if(n.pt(m,d,s[0])&&n.pt(m,d,s[1])&&t.b[d]==2)return 2;if(n.pt(m,d,s[0])&&n.pt(m,d,s[1])&&t.b[d]!=0)return 1;}}}if(U.wj(m.k,e).length==5)for(g=1,P=U.wj(m.j,e),r=0,i=P.length;r<i;++r){if(A=P[r],n.up(t,A)+n.np(t,A)==t.e[A]+n.np(t,A)&&g>0){--g;continue;}if(n.up(t,A)+n.np(t,A)!=t.e[A]+n.np(t,A)+1)return 1;}return 0;},i1.zF=function(t,e,A){return i1.AF(t,e,e.D[0][A],e.D[1][A]);},i1.AF=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b;for(r=e.a[A],i=e.a[P],L=n.Wo(e,A,P),s=true,b=(n.aw(e,7),e.p),h=0;h<b.j.a.length&&s;h++)for(u=U.wj(b.k,h),o=0,Q=u.length;o<Q;++o)if(D=u[o],D==L&&e.b[h]==0){s=false;break;}return n.em(e,L)==1&&s&&(At.ZD(t.b,r)&&At.ZD(t.b,i)||At.dE(t.b,r)&&At.dE(t.b,i))?1:0;},i1.BF=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;for(u=i1.CF(e),D=z(l.b7,jt,6,u.length,15,1),P=0;P<u.length;P++){if(d=e.a[P],b=u[P],m=At.tE(t.e,d),A=At.$D(t.b,d),c=0,g=0,j.Math.abs(m)<wA)for(Q=0;Q<e.e[P];Q++)s=e.i[P][Q],u[s]<0&&(b+=u[s]/(2*e.e[s]));if(d==62)for(L=0;L<e.e[P];L++)s=e.i[P][L],u[s]>0&&(b-=u[s]/2);for(o=0;o<e.e[P];o++)s=e.i[P][o],r=n.Wo(e,P,s),h=e.a[s],i=i1.AF(t,e,e.D[0][r],e.D[1][r]),g+=At.uE(t.e,i,d,h),c+=u[s];D[P]=(1-A*m)*b+m*c+g;}return D;},i1.CF=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N;D=z(l.b7,jt,6,t.q,15,1);t:for(i=0;i<D.length;i++){switch(N=t.a[i],D[i]=0,N){case 32:case 72:for(Q=0;Q<t.e[i];Q++){for(m=t.i[i][Q],M=t.a[m],g=0,d=0,s=0;s<t.e[m];s++)C=t.i[m][s],t.C[C]==7&&t.e[C]+n.np(t,C)==2&&!(C<t.f&&n.nt(t.p,C))&&++g,(t.C[C]==8||t.C[C]==16)&&t.e[C]+n.np(t,C)==1&&++d;if(t.C[m]==16&&d==2&&g==1&&(g=0),t.C[m]==6&&d>0){D[i]=d==1?-1:-(d-1)/d;continue t;}if(M==45&&d==3){D[i]=-0.3333333333333333;continue t;}if(M==25&&d>0){D[i]=d==1?0:-(d-1)/d;continue t;}if(M==18&&d>0){D[i]=g+d==2?0:-(g+d-2)/d;continue t;}if(M==73&&d>0){D[i]=d==1?0:-(d-1)/d;continue t;}if(M==77&&d>0){D[i]=-1/d;continue t;}}break;case 76:for(O=0,K=(n.aw(t,7),t.p),V=0;V<K.j.a.length;V++)if(n.pt(K,V,i)){O=V;break;}if(O<K.j.a.length){for(c=0,A=U.wj(K.j,O),P=0,r=A.length;P<r;++P)e=A[P],e>-1&&t.a[e]==76&&++c;if(c>0){D[i]=-1/c;continue t;}}break;case 55:case 56:case 81:for(D[i]=t.s[i],b=1,S=0,u=z(l.zeb,ht,6,t.q,16,1),u[i]=true;b>S;)for(S=b,L=0;L<t.q;L++)if(u[L]){for(s=0;s<t.e[L];s++)if(m=t.i[L][s],M=t.a[m],!(M!=57&&M!=80))for(h=0;h<t.e[m];h++)C=t.i[m][h],v=t.a[C],!(v!=55&&v!=56&&v!=81)&&(u[C]||(u[C]=true,D[i]+=t.s[C],++b));}D[i]/=b;continue t;case 61:for(o=0;o<t.e[i];o++)if(m=t.i[i][o],t.a[m]==42){D[i]=1;continue t;}continue t;case 34:case 49:case 51:case 54:case 58:case 92:case 93:case 94:case 97:D[i]=1;continue t;case 87:case 95:case 96:case 98:case 99:D[i]=2;continue t;case 88:D[i]=3;continue t;case 35:case 62:case 89:case 90:case 91:D[i]=-1;continue t;}D[i]=0;}return D;},i1.DF=function(t,e,A,P,r,i){var D,u,o,Q,L,s;return Q=n.Wo(e,P,r),D=i1.AF(t,e,A,P),u=i1.AF(t,e,P,r),o=i1.AF(t,e,r,i),s=u,u==0&&n.em(e,Q)==1&&(D==1||o==1)&&(s=2),L=i1.EF(e,A,P,r,i),L==4&&n.Wo(e,A,r)==-1&&n.Wo(e,P,i)==-1?40+s:L==5&&(e.a[A]==1||e.a[P]==1||e.a[r]==1||e.a[i]==1)?50+s:s;},i1.EF=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h;if(n.Wo(t,e,A)==-1||n.Wo(t,A,P)==-1||n.Wo(t,P,r)==-1)return 0;if(n.Wo(t,r,e)>=0)return 4;for(s=(n.aw(t,7),t.p),h=new U.wtb(),U.vtb(h,T.uib(e)),U.vtb(h,T.uib(A)),U.vtb(h,T.uib(P)),U.vtb(h,T.uib(r)),Q=0;Q<s.j.a.length;Q++)if(U.wj(s.k,Q).length==5){for(L=new U.wtb(),D=U.wj(s.j,Q),u=0,o=D.length;u<o;++u)i=D[u],U.vtb(L,T.uib(i));if(U.Ii(L,h))return 5;}return 0;},x(415,1,{}),l.i9=X(415),A1.GF=function(){A1.GF=o1,A1.FF=(T.Ujb(),p1);},A1.HF=function(t){},A1.IF=function(t){var e,A,P;for(e=0,A=0,P=0;P<t.g.a.length+t.f.a.length;P++)e+=n.Tl(P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length))*(P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length)).r,A+=(P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length)).r;return A!=0?A/e:1;},A1.JF=function(t,e){A1.GF();var A,P,r,i;A1.HF(this),P=new tt.zH(t);try{for(i=new Rt.xgb(),Rt.wgb(i,"$RXN"+A1.FF),T.Fjb(i.a,e??""),Rt.wgb(i,A1.FF+(""+A1.FF)),Rt.wgb(i,hP+tt.UH(t,!0,27)),Rt.wgb(i,A1.FF),Rt.wgb(i,"  "+P.g.a.length+"  "+P.f.a.length+A1.FF),r=A1.IF(P),A=0;A<P.g.a.length+P.f.a.length;A++)Rt.wgb(i,"$MOL"+A1.FF),n.Zr(new n.as(A<P.g.a.length?U.wj(P.g,A):U.wj(P.f,A-P.g.a.length),r,null),i);this.a=i.a;}catch(D){if(D=H1(D),Jt(D,19))T.Ujb();else throw Ut(D);}},x(279,1,{},A1.JF),y.a=null,l.j9=X(279),A1.KF=function(t){var e,A;return e=new tt.yH(),A=new Rt.igb(new Rt.tgb(t)),A1.LF(e,A),e;},A1.LF=function(t,e){var A,P;if(P=Rt.hgb(e),A=false,P==null||!T.ejb(w.Xzb(P).substr(0,4),"$RXN"))throw Ut(new T.bC("'$RXN' tag not found"));return T.ejb(P,bi)?A=A1.NF(t,e,false):A=A1.MF(t,e,false),A;},A1.MF=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;if(Q=T.wjb(Rt.hgb(e)),w.Xzb(Q).length!=0&&(t.e=Q),Rt.hgb(e),P=Rt.hgb(e),!A&&T.ejb(w.Xzb(P).substr(0,13),hP)&&(r=(w.Szb(13,w.Xzb(P).length+1),w.Xzb(P).substr(13)),tt.TH(r,true,t)))return  true;for(c=Rt.hgb(e),b=T.Qhb(T.wjb((w.Rzb(0,3,w.Xzb(c).length),w.Xzb(c).substr(0,3))),P1,St),s=T.Qhb(T.wjb((w.Rzb(3,6,w.Xzb(c).length),w.Xzb(c).substr(3,3))),P1,St),u=new n.Bs(),D=0;D<b;D++){if(c=Rt.hgb(e),c==null||!T.ejb(w.Xzb(c).substr(0,4),"$MOL"))throw Ut(new T.bC(ci));h=new n.sw(),o=new T.Hjb();do c=Rt.hgb(e),o.a+=""+c,o.a+=p1;while(c!=null&&!T.ejb(w.Xzb(c).substr(0,6),me));if(c==null)throw Ut(new T.bC(wi));n.ws(u,h,new Rt.igb(new Rt.tgb(o.a))),U.rj(t.g,h),t.d=-1;}for(i=0;i<s;i++){if(c=Rt.hgb(e),c==null||!T.ejb(w.Xzb(c).substr(0,4),"$MOL"))throw Ut(new T.bC(ci));L=new n.sw(),o=new T.Hjb();do c=Rt.hgb(e),o.a+=""+c,o.a+=p1;while(c!=null&&!T.ejb(w.Xzb(c).substr(0,6),me));if(c==null)throw Ut(new T.bC(wi));n.ws(u,L,new Rt.igb(new Rt.tgb(o.a))),U.rj(t.f,L),t.d=-1;}return  true;},A1.NF=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;if(Q=T.wjb(Rt.hgb(e)),w.Xzb(Q).length!=0&&(t.e=Q),Rt.hgb(e),P=Rt.hgb(e),!A&&T.ejb(w.Xzb(P).substr(0,13),hP)&&(r=(w.Szb(13,w.Xzb(P).length+1),w.Xzb(P).substr(13)),tt.TH(r,true,t)))return  true;if(c=Rt.hgb(e),D=new n.Bs(),c!=null&&T.ejb(w.Xzb(c).substr(0,13),"M  V30 COUNTS")){if(b=T.wjb((w.Szb(13,w.Xzb(c).length+1),w.Xzb(c).substr(13))),L=T.sjb(b," "),h=T.Qhb(L[0],P1,St),s=T.Qhb(L[1],P1,St),h>0){if(c=Rt.hgb(e),T.ejb(gi,c))for(i=0;i<h;i++){u=new n.sw(),o=new T.Hjb(),o.a+=zi;do c=Rt.hgb(e),o.a+=""+c,o.a+=p1;while(c!=null&&!T.ejb(w.Xzb(c).substr(0,15),Lr));n.ws(D,u,new Rt.igb(new Rt.tgb(o.a))),U.rj(t.g,u),t.d=-1;}c=Rt.hgb(e);}if(s>0&&(c=Rt.hgb(e),T.ejb(di,c))){for(i=0;i<s;i++){u=new n.sw(),o=new T.Hjb(),o.a+=zi;do c=Rt.hgb(e),o.a+=""+c,o.a+=p1;while(c!=null&&!T.ejb(w.Xzb(c).substr(0,15),Lr));n.ws(D,u,new Rt.igb(new Rt.tgb(o.a))),U.rj(t.f,u),t.d=-1;}c=Rt.hgb(e);}return  true;}return  false;},A1.PF=function(){A1.PF=o1,A1.OF=(T.Ujb(),p1);},A1.QF=function(t){},A1.RF=function(t){var e,A,P;for(e=0,A=0,P=0;P<t.g.a.length+t.f.a.length;P++)e+=n.Tl(P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length))*(P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length)).r,A+=(P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length)).r;return A!=0?A/e:1;},A1.SF=function(t,e){A1.PF();var A,P,r,i,D,u;A1.QF(this),i=new tt.zH(t);try{if(u=new Rt.xgb(),Rt.wgb(u,bi+A1.OF),T.Fjb(u.a,e??""),Rt.wgb(u,A1.OF+(""+A1.OF)),Rt.wgb(u,hP+tt.UH(t,!0,11)),Rt.wgb(u,A1.OF),r=i.g.a.length,P=i.f.a.length,Rt.wgb(u,Y8+r+" "+P+A1.OF),D=A1.RF(i),r>0){for(Rt.wgb(u,gi+A1.OF),A=0;A<i.g.a.length;A++)Rt.wgb(u,n.Ns(U.wj(i.g,A),D));Rt.wgb(u,"M  V30 END REACTANT"+A1.OF);}if(P>0){for(Rt.wgb(u,di+A1.OF),A=0;A<i.f.a.length;A++)Rt.wgb(u,n.Ns(U.wj(i.f,A),D));Rt.wgb(u,"M  V30 END PRODUCT"+A1.OF);}Rt.wgb(u,me+A1.OF),this.a=u.a;}catch(o){if(o=H1(o),Jt(o,19))T.Ujb();else throw Ut(o);}},x(280,1,{},A1.SF),y.a=null,l.k9=X(280),A1.UF=function(){A1.UF=o1,A1.TF=F(k(l.Sbb,1),st,2,6,["Idorsia No","Actelion No","ID","IDNUMBER","COMPOUND_ID","NAME","COMPND"]);},A1.VF=function(t){var e,A,P,r,i,D,u,o,Q;if(!t.i)return  false;T.Agb(t.g,0),T.Agb(t.b,0),t.f=null,Q=false,A=-1,P=null,t.c=t.d==null?null:z(l.Sbb,st,2,t.d.length,6,1),t.e=-1;do{try{if(o=Rt.hgb(t.i),o==null)return T.Agb(t.g,0),!1;}catch(L){if(L=H1(L),Jt(L,76))return T.Agb(t.g,0),false;throw Ut(L);}if(Q)T.Njb(t.b,o),T.Jjb(t.b,10);else if(T.ejb(w.Xzb(o).substr(0,1),">"))Q=true,T.Njb(t.g,me),T.Jjb(t.g,10),T.Njb(t.b,o),T.Jjb(t.b,10);else {T.Njb(t.g,o),T.Jjb(t.g,10),T.ejb(w.Xzb(o).substr(0,6),me)&&(Q=true);continue;}if(t.d!=null)if(w.Xzb(o).length==0)A=-1;else if(A==-1){if(P=A1.XF(o),P!=null){for(A=-1,e=0;e<t.d.length;e++)if(T.ejb(P,t.d[e])){A=e;break;}if(t.e==-1){for(i=A1.TF,D=0,u=i.length;D<u;++D)if(r=i[D],T.ejb(P,r)){t.e=A;break;}}}}else t.c[A]==null?t.c[A]=o:t.c[A]=T.ajb(T.ajb(t.c[A],p1),o);}while(!T.ejb(w.Xzb(o).substr(0,4),QA));return  true;},A1.WF=function(t,e){var A,P,r,i;for(i=0,P=new n.qx();i<e;){try{r=Rt.hgb(t.i);}catch(D){if(D=H1(D),Jt(D,76))break;throw Ut(D);}if(r==null)break;T.ejb(w.Xzb(r).substr(0,4),QA)&&++i,T.ejb(w.Xzb(r).substr(0,1),">")&&(A=A1.XF(r),A!=null&&n.px(P,A));}t.d=U.Kj(P.b,z(l.Sbb,st,2,0,6,1));},A1.XF=function(t){var e,A,P,r;if(w.Xzb(t).length==0||(w.Szb(0,w.Xzb(t).length),w.Xzb(t).charCodeAt(0)!=62))return null;for(P=1,r=0,e=0;P<w.Xzb(t).length;){if(w.Szb(P,w.Xzb(t).length),w.Xzb(t).charCodeAt(P)==60){if(r!=0)return null;r=P;}else if(w.Szb(P,w.Xzb(t).length),w.Xzb(t).charCodeAt(P)==62){if(e!=0)return null;e=P;}++P;}if(r!=0&&r<e)return w.Rzb(r+1,e,w.Xzb(t).length),w.Xzb(t).substr(r+1,e-(r+1));if(P=w.Xzb(t).indexOf("DT",1),P==-1)return null;for(A=P+2;w.Xzb(t).length>A&&ee((w.Szb(A,w.Xzb(t).length),w.Xzb(t).charCodeAt(A)));)++A;return A==P+2?null:(w.Rzb(P,A,w.Xzb(t).length),w.Xzb(t).substr(P,A-P));},A1.YF=function(t,e){return t.c==null?null:t.c[e];},A1.ZF=function(t){return t.d==null&&A1.WF(t,10240),t.d;},A1.$F=function(t,e){return t.d==null&&A1.WF(t,e),t.d;},A1._F=function(t){var e,A;return t.f||(A=new n.Bs(),t.a&&(A.a=true),t.f=n.es(A,(e=t.g.a,e)),t.f&&(t.f.P==null||w.Xzb(t.f.P).length==0)&&n.Tn(t.f,t.e!=-1&&t.c!=null?t.c[t.e]:t.f?t.f.P:null)),t.f;},A1.aG=function(t,e){A1.UF(),this.d=e,this.i=new Rt.igb(t),this.g=new T.Qjb(),this.b=new T.Qjb();},x(281,415,{},A1.aG),y.a=false,y.e=0,l.l9=X(281),nt.eG=function(){nt.eG=o1,nt.bG=F(k(l.e7,1),n0,6,14,[524290,524292,589826,589827,589828,589829,589832,589833,589837,589838,598019,598020,598024,606211,688130,688131,688132,688137,688142,696324,819202,819203,819204,819205,819214,827396,1048580,1114114,1114115,1114116,1114120,1212420,1220611,1343492,1343496,1638402,1638403,17367042,17367043,17367044,17367045,17367048,17367049,17367053,17367054,17375236,17465346,17465348,17465353,17473540,1208483842,1208483848,1208483912,1208549378,1208549379,1208549380,1208549384,1208549388,1208549442,1208549443,1208549444,1208549448,1208557571,1208565763,1208565827,1409875970,1409875971,1409875972,1409875976,1409876034,1409876035,1409876036,1409876040,1409884163,1409974274,1409974280,1409974338,1678245892,1678311426,1678311427,1678311428,1678311432,1678311490,1678311491,1678311492,1678311496,1678409730,1678409731,1678409732,1678409794,1678540802,1678540808,1678540810,1678540866,2282291202,2282291203,2282291266,2282291267,2282389506,2282389507,2282389570,2282389571,2282389572,2282520578,2282520579,2282520642,2282520643,2282537027,2483617794,2483617795,2483617858,2483617859,2483617860,2483617864,2483716098,2483716099,2483716162,2483716163,2484150275,2484248579,2752675842,3356033026,3356139523,3557359618,3557457922,3557588994,34510798914,34510798915,34510798916,34510798920,34510807107,34510815299,34712125506,34712125507,34712125508,34712125512,34712223810,34712223811,34712223812,34712223816,34980560962,34980560963,34980659266,34980659267,34980659268,35568287746,35568287747,35568287748,35568287752,35568287756,35568287810,35568287811,35568287812,35568287816,35568287820,35568386050,35568386051,35568386052,35568386114,35568386120,35568517122,35568517123,35568517124,35568517128,35568517186,35568517187,35568517188,35568517192,35568812034,35568812035,35568812098,35568812099,35568910338,35568910339,35568910403,35569336322,35569434626,35585064962,35585064963,35585064964,35585064968,35585065026,35585065027,35585065028,35585065032,35769614338,35769614340,35769614402,35769712642,35770138626,35770236930,35786391554,35786391555,2475109646403,ji,Bi,2475109711938,2475109711939,2475109720067,2475109720131,2475109728259,2475109728323,mi,ki,2887426572354,2887426572355,Fi,vi,2887627898946,pi,2887627997250,Ri,Ci,3437182386242,3437182386243,Hi,3437383712771,3437383712834,Mi,Si,qi,Vi,4674132967490,4674132967491,4674132975619,Ei,4674334294082,4674334294083,4674334302211,4674334302275,Ni,4674334392386,Ii,4674602729538,4674602827842,xi,4674602958914,Oi,Xi,5086449827906,5086449827907,5086449827912,Gi,5086651154498,5086651162627,Ki,Yi,5086651252802,5086651252808,5086651260931,Ji,5086919589954,Zi,5086919688258,5086919819330,5636205641794,_i,5636406968386,Wi,5636407066690,5636675403842,$i,5636675502146,{l:589890,m:73760,h:4},{l:589891,m:73760,h:4},{l:598083,m:73760,h:4},{l:688194,m:73760,h:4},{l:688195,m:73760,h:4},{l:696387,m:73760,h:4},{l:819266,m:73760,h:4},{l:65602,m:73764,h:4},{l:65603,m:73764,h:4},{l:589890,m:172064,h:4},{l:589891,m:172064,h:4},{l:598083,m:172064,h:4},{l:688194,m:172064,h:4},{l:688195,m:172064,h:4},{l:696387,m:172064,h:4},{l:819266,m:172064,h:4},{l:65602,m:172068,h:4},{l:65603,m:172068,h:4},{l:589890,m:172112,h:4},{l:589891,m:172112,h:4},{l:688194,m:172112,h:4},{l:819266,m:172112,h:4},{l:65602,m:172116,h:4},{l:65608,m:172116,h:4},{l:163906,m:172116,h:4},{l:589890,m:303136,h:4},{l:688194,m:303136,h:4},{l:819266,m:303136,h:4},{l:65602,m:303140,h:4},{l:589890,m:303184,h:4},{l:688194,m:303184,h:4},{l:819266,m:303184,h:4},{l:65602,m:303188,h:4},{l:65602,m:303252,h:4},ai,tf,{l:589890,m:590112,h:4},{l:589891,m:590112,h:4},ef,nf,{l:589890,m:590160,h:4},{l:589891,m:590160,h:4},Af,{l:688194,m:590160,h:4},Pf,rf,{l:589890,m:590224,h:4},{l:589891,m:590224,h:4},{l:688131,m:590224,h:4},{l:688194,m:590224,h:4},ff,Df,{l:589890,m:590368,h:4},uf,{l:688194,m:590368,h:4},{l:696323,m:590368,h:4},of,lf,{l:819266,m:590368,h:4},Qf,Lf,{l:589890,m:590416,h:4},{l:589891,m:590416,h:4},{l:589896,m:590416,h:4},yf,{l:688136,m:590416,h:4},{l:688194,m:590416,h:4},{l:696323,m:590416,h:4},Uf,{l:819266,m:590416,h:4},{l:688130,m:590480,h:4},{l:688194,m:590480,h:4},{l:65602,m:598052,h:4},{l:65603,m:598052,h:4},{l:65602,m:598100,h:4},{l:65603,m:598100,h:4},{l:73795,m:598100,h:4},{l:163906,m:598100,h:4},{l:163907,m:598100,h:4},{l:65602,m:598164,h:4},{l:163906,m:598164,h:4},sf,hf,{l:589890,m:598304,h:4},{l:589891,m:598304,h:4},Tf,{l:688194,m:598304,h:4},bf,{l:1114178,m:598304,h:4},cf,{l:1212424,m:598304,h:4},{l:1212482,m:598304,h:4},wf,gf,{l:589890,m:688416,h:4},zf,{l:589890,m:688464,h:4},{l:589890,m:688528,h:4},df,jf,{l:688130,m:688784,h:4},{l:65602,m:696356,h:4},{l:65602,m:696404,h:4},{l:163906,m:696404,h:4},{l:589826,m:696608,h:4},{l:1114178,m:696608,h:4},{l:1212418,m:696608,h:4},{l:589826,m:696612,h:4},{l:524355,m:590112,h:288},{l:589825,m:590112,h:288},Bf,{l:589890,m:590112,h:288},{l:598019,m:590112,h:288},{l:598083,m:590112,h:288},mf,{l:589890,m:590112,h:336},{l:598019,m:590112,h:336},{l:598083,m:590112,h:336},kf,{l:589890,m:688416,h:336},Ff,{l:589890,m:688464,h:336},vf,pf,{l:589827,m:590112,h:400},{l:589890,m:590112,h:400},{l:589891,m:590112,h:400},Rf,Cf,Hf,Mf,{l:589890,m:819488,h:400},Sf,qf,Vf,Ef,{l:589827,m:819488,h:544},Nf,If,xf,{l:688199,m:688464,h:592},Of,Xf,Gf,{l:688202,m:819600,h:592},Kf,{l:589896,m:1212704,h:592},Yf,Jf,{l:688200,m:1212752,h:592},Zf,{l:688135,m:688464,h:656},{l:688135,m:819536,h:656},{l:589891,m:65936,h:8228},_f,{l:589890,m:590112,h:8480},{l:598019,m:590112,h:8480},Wf,{l:589890,m:688416,h:8480},$f,{l:589890,m:688464,h:8480},af,tD,eD,nD,{l:589896,m:1213008,h:8480},AD,{l:688200,m:1213008,h:8480},{l:688135,m:1343824,h:8480},PD,{l:589890,m:590112,h:8484},rD,{l:589890,m:590160,h:8484},{l:1212424,m:590416,h:8484},{l:1212488,m:590416,h:8484},iD,{l:688194,m:598304,h:8484},{l:589826,m:590112,h:8528},{l:589890,m:590112,h:8528}]),nt.cG=F(k(l.c7,1),w0,6,15,[.6966999769210815,0,.4885999858379364,-0.47269999980926514,-0.07490000128746033,fD,.273499995470047,.5699999928474426,.7009999752044678,.9534000158309937,-3.6435000896453857,-2.150899887084961,.4975000023841858,-2.19950008392334,-0.2809000015258789,-0.8259999752044678,-0.1784999966621399,-1.620300054550171,-1.0959999561309814,-0.3619999885559082,.13950000703334808,-0.29750001430511475,-1.2907999753952026,1.0161999464035034,DD,-2.5383999347686768,TP,.4291999936103821,-0.5824000239372253,-0.1834000051021576,.1306000053882599,-0.5015000104904175,fD,-0.5257999897003174,.4244000017642975,-0.16099999845027924,-0.2777999937534332,.5110999941825867,-0.435699999332428,-0.10409999638795853,.3424000144004822,-0.061500001698732376,.6035000085830688,.7226999998092651,.43459999561309814,-1.6821000576019287,-0.3310000002384186,-0.49799999594688416,uD,-0.3650999963283539,.45969998836517334,.3384000062942505,.6632999777793884,.4544000029563904,.15970000624656677,.633899986743927,.35040000081062317,.04490000009536743,.34200000762939453,.26109999418258667,.40459999442100525,.5218999981880188,-1.065000057220459,-1.2314000129699707,-1.802299976348877,-0.36320000886917114,-0.4108000099658966,.30570000410079956,-0.14560000598430634,-0.27129998803138733,-0.5192999839782715,.45260000228881836,.5539000034332275,.8374000191688538,-0.7070000171661377,-0.48809999227523804,-0.4099999964237213,0,.14790000021457672,.3447999954223633,.42980000376701355,.5579000115394592,-0.1264999955892563,-0.042500000447034836,.07670000195503235,.6635000109672546,-0.38119998574256897,-0.8367999792098999,1.0286999940872192,-0.10209999978542328,.3587000072002411,-0.5945000052452087,.16920000314712524,-0.121799997985363,.32829999923706055,.22390000522136688,.20430000126361847,.05900000035762787,-0.48350000381469727,.6165000200271606,-0.4011000096797943,.5577999949455261,-0.21639999747276306,-0.017500000074505806,.29809999465942383,.10999999940395355,.27149999141693115,-0.2994999885559082,-0.46700000762939453,.1565999984741211,.046799998730421066,-0.13210000097751617,1.3686000108718872,0,-0.4115999937057495,1.0185999870300293,-0.3935000002384186,.5223000049591064,.3684999942779541,.25769999623298645,1.5192999839782715,.2705000042915344,.3790999948978424,.012000000104308128,-0.33970001339912415,.14830000698566437,.2766000032424927,.35929998755455017,.7714999914169312,.3149999976158142,-1.618499994277954,.18889999389648438,-0.2651999890804291,-0.09650000184774399,.420199990272522,.18709999322891235,-0.3684000074863434,-0.07779999822378159,.8942999839782715,.3693999946117401,.28790000081062317,.4489000141620636,-0.26010000705718994,.4771000146865845,.1923000067472458,.43810001015663147,.16949999332427979,.45249998569488525,.3352000117301941,.1582999974489212,.4036000072956085,-0.04800000041723251,.5023000240325928,-0.26489999890327454,.76910001039505,-0.35519999265670776,1.0300999879837036,-0.11410000175237656,-0.5932000279426575,.17489999532699585,.13130000233650208,-0.18039999902248383,.399399995803833,.22910000383853912,.31690001487731934,.35989999771118164,-0.0038999998942017555,-0.2955999970436096,.4408999979496002,-0.16089999675750732,.3774999976158142,-0.13459999859333038,.2838999927043915,.5128999948501587,.1265999972820282,.4293999969959259,.28060001134872437,.49070000648498535,oD,.219200000166893,.15649999678134918,.6934999823570251,.3617999851703644,.6735000014305115,.5777999758720398,-0.5636000037193298,.5569000244140625,.30379998683929443,-0.32760000228881836,-0.6991999745368958,.010300000198185444,-0.4659000039100647,lD,TP,.18410000205039978,.707099974155426,.12269999831914902,.7949000000953674,-0.6592000126838684,-1.3148000240325928,-0.4066999852657318,-0.1316000074148178,-0.4925000071525574,-0.09290000051259995,-0.4352000057697296,-0.2206999957561493,-0.9959999918937683,-0.723800003528595,-0.5468999743461609,-1.2939000129699707,-0.01360000018030405,.065700002014637,.7189000248908997,.05700000002980232,.661899983882904,-0.6381000280380249,-0.8072999715805054,.23549999296665192,.30480000376701355,-0.019899999722838402,-0.07519999891519547,.44609999656677246,.1559000015258789,1.1167999505996704,-1.8039000034332275,.23649999499320984,-0.22059999406337738,.4480000138282776,-1.1339999437332153,-0.5652999877929688,-0.40529999136924744,-0.13609999418258667,.2198999971151352,.053599998354911804,-0.020999999716877937,.6984999775886536,.9642999768257141,-0.41519999504089355,-1.0369000434875488,-0.18299999833106995,.5882999897003174,-0.29179999232292175,-0.5293999910354614,-0.6541000008583069,.9473000168800354,-0.19059999287128448,-0.8483999967575073,-0.3456999957561493,.9541000127792358,1.4230999946594238,-0.7924000024795532,QD,LD,-0.2596000134944916,.8381999731063843,-0.4415999948978424,-0.37040001153945923,-0.7487000226974487,-0.10790000110864639,-0.29919999837875366,-0.3276999890804291,.025100000202655792,-0.9187999963760376,.10939999669790268,.8230999708175659,-3.233299970626831,.03500000014901161,lD,yD,.2791000008583069,.3206000030040741,.5662000179290771,-0.3783999979496002,.4032000005245209,-1.794800043106079,-0.15539999306201935,.3785000145435333,.05339999869465828,-0.16529999673366547,-0.09870000183582306,-0.10050000250339508,-0.6460999846458435,.8034999966621399,-0.24050000309944153,-0.12380000203847885,-0.35760000348091125,.09610000252723694,-0.6401000022888184,.2029000073671341,.23589999973773956,.4950999915599823,.19210000336170197,-0.37450000643730164,.34630000591278076,.289900004863739,-0.15330000221729279,-0.4169999957084656,.37700000405311584,.6998000144958496,.593999981880188,.5911999940872192,-0.5570999979972839,.023800000548362732,-0.2475000023841858,.030700000002980232,-0.38749998807907104,-0.7437000274658203,.5144000053405762,.00570000009611249,.765500009059906,.1720000058412552,-2.5624001026153564,-0.30660000443458557,.36469998955726624,.17270000278949738,-0.03290000185370445,-0.18930000066757202,.07020000368356705,-1.2453999519348145,.14959999918937683,DD,.4146000146865845,-0.2667999863624573,-0.11060000211000443,.03620000183582306,-0.3188999891281128,-0.7278000116348267,-0.08940000087022781,-0.22769999504089355,-0.2393999993801117,1.011199951171875,-0.2962000072002411,.7775999903678894,.2944999933242798,-0.22339999675750732,.27639999985694885,.8011000156402588,-0.17440000176429749,.15809999406337738,-1.7552000284194946,-0.384799987077713,.5993000268936157,.5267999768257141,-0.04170000180602074,.4733000099658966,-0.3400999903678894,-0.14499999582767487,.7088000178337097,-0.13179999589920044,.04259999841451645,-0.5027999877929688,.3831999897956848,-0.011800000444054604,-0.4357999861240387,.3749000132083893,-0.12030000239610672,-0.5648000240325928,-0.1972000002861023,-0.8769000172615051,-0.3675000071525574,-0.2003999948501587,-0.6069999933242798,-0.18569999933242798,.3467999994754791,-0.36239999532699585,.5357999801635742,-0.3700999915599823,.13359999656677246,.9545000195503235,.11389999836683273,-0.16990000009536743,TP,.2890999913215637,.2612999975681305,-0.03440000116825104,-1.9498000144958496,-2.024899959564209,-0.6004999876022339,-0.6258000135421753,-1.2348999977111816,.328000009059906,-0.54339998960495,-0.7712000012397766,-0.9057000279426575,-0.16680000722408295,-0.9904999732971191,-0.03720000013709068,-1.1638000011444092,.12620000541210175,-0.5248000025749207,-0.15379999577999115,-0.36820000410079956,.3249000012874603,.06499999761581421,.051100000739097595,-0.46070000529289246,.22310000658035278,.28220000863075256,.1396999955177307,-0.49380001425743103,.39480000734329224,-0.4074999988079071,-0.6410999894142151,-0.009100000374019146,-0.13330000638961792,-0.5192000269889832,-0.16609999537467957,TP,-0.07069999724626541,.4805999994277954,.38280001282691956,.22290000319480896,.6159999966621399,.33709999918937683,.188400000333786,.13809999823570251,uD,.2833000123500824,-0.1225999966263771,-3.9189000129699707,-0.459199994802475,-0.3434999883174896,-0.6654000282287598,-0.5055999755859375,-0.863099992275238,.15360000729560852,-0.6427000164985657,-0.08839999884366989,-0.0471000000834465,.11060000211000443,.382099986076355,-0.23919999599456787,-0.4050999879837036,.08910000324249268,-0.6972000002861023,-0.4699000120162964,.09220000356435776,.08060000091791153,-0.6773999929428101,-0.062199998646974564,-0.9300000071525574,.13369999825954437]);},nt.fG=function(t){var e,A,P,r,i;for(A=0,n.kq(t),n.aw(t,7),e=0;e<t.f;e++)try{i=n.ef(t,e,6241),r=(P=lt.a4(nt.dG,T.Jib(i)),P<0?-1:P),r!=-1&&(A+=nt.cG[r]);}catch(D){if(D=H1(D),!Jt(D,19))throw Ut(D);}return A;},nt.gG=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g;if(r=new nt.vG(),U.rj(r.a,new nt.sG("cLogP Values are estimated applying an atom-type based increment system.",2)),U.rj(r.a,new nt.sG(UD,2)),U.rj(r.a,new nt.sG(Vr,2)),t){for(n.kq(t),n.aw(t,7),u=0,P=new U.awb(),o=new o0.Mmb("#0.000"),e=0;e<t.f;e++)try{A=n.ef(t,e,6241),g=U.jpb(P,T.Jib(A)),g?U.Uvb(P,T.Jib(A),T.uib(g.a+1)):U.Uvb(P,T.Jib(A),T.uib(1));}catch(d){if(d=H1(d),Jt(d,19))++u;else throw Ut(d);}for(u!=0&&U.rj(r.a,new nt.sG("Warning: "+u+" atom type(s) could not be determined.",2)),c=(D=new U.kwb(new U.qwb(new U.xpb(P).a).b),new U.Gpb(D));U.fob(c.a.a);)b=(i=U.iwb(c.a),i.fd()),L=lt.a4(nt.dG,b),(L<0?-1:L)!=-1?nt.uG(r,(h=b,U.anb(U.Pvb(P,h))+" * "+o0.Jmb(o,nt.cG[Q=lt.a4(nt.dG,b),Q<0?-1:Q])+" AtomType: 0x"+T.Hib(b.a)),2):nt.uG(r,"Warning: For atom type 0x"+T.Hib(b.a)+" ("+(s=b,U.anb(U.Pvb(P,s)))+" times found) is no increment available.",2);}return r;},nt.hG=function(){nt.eG();var t,e,A,P;if(!nt.dG&&!nt.dG)for(nt.dG=new lt.d4(),e=nt.bG,A=0,P=e.length;A<P;++A)t=T.Jib(e[A]),lt.Z3(nt.dG,t);},x(213,1,{},nt.hG),l.m9=X(213),nt.iG=function(t,e,A,P,r){var i,D,u,o,Q,L;for(i=1/(1+j.Math.exp(t-5)),L=1-1/(1+j.Math.exp(e+5)),Q=1/(1+j.Math.exp(.012*A-6)),D=1-1/(1+j.Math.exp(P)),u=(.5+i/2)*(.5+L/2)*(.5+Q/2)*(.5+D/2),o=0;r!=null&&o<r.length;o++)r[o]==2?u*=.8:r[o]==3&&(u*=.6);return u;},nt.lG=function(t,e){var A,P,r,i,D,u,o,Q,L;if(A=new nt.vG(),!nt.kG)return U.rj(A.a,new nt.sG(sD,2)),-999;for(U.rj(A.a,new nt.sG("Found sub-structure fragments and their contributions:",2)),U.rj(A.a,new nt.sG("(yellow atoms carry at least one more substituent)",2)),Q=0,o=0,i=0,L=new n.iu(1),r=new n.sw(),D=0;D<nt.jG.a.a.length;D++)n.Sq(new n.$q(false),r,nt.oG(nt.jG,D)),n.bu(L,e),n.$t(L,r),n.Ot(L,1,L.c)>0&&(u=nt.pG(nt.jG,D),u<-1?Q+=u:(o+=u,++i),nt.uG(A,nt.oG(nt.jG,D),1),U.rj(A.a,new nt.sG(""+u,3)));return i==0?-1:(P=Q+o/j.Math.sqrt(i),P=P+.0625*(i-40),t.a=A,P);},nt.mG=function(t){var e,A,P,r,i,D;if(!nt.kG)return sD;for(i=0,P=0,D=new n.iu(1),A=new n.sw(),r=0;r<nt.jG.a.a.length;r++)n.Sq(new n.$q(false),A,nt.oG(nt.jG,r)),n.bu(D,t),n.$t(D,A),n.Ot(D,1,D.c)>0&&(i+=nt.pG(nt.jG,r),++P);return e=P==0?-1:i/j.Math.sqrt(P),e+"	"+P+"	"+t.f;},nt.nG=function(){if(!nt.kG)try{nt.jG=new nt.qG(),nt.kG=!0;}catch(t){if(t=H1(t),Jt(t,19))T.Ujb();else throw Ut(t);}},x(273,1,{},nt.nG),nt.kG=false,l.n9=X(273),nt.oG=function(t,e){return U.wj(t.a,e).a;},nt.pG=function(t,e){return U.wj(t.a,e).b;},nt.qG=function(){var t,e,A,P,r,i;for(i=new Rt.igb(new Rt.ngb((A=_t.hAb("/resources/druglikenessNoIndex.txt"),V1.Hmb(),A))),this.a=new U.Lj();r=Rt.hgb(i),r!=null;){if(P=T.ljb(r,Z1(9)),P==-1)throw Ut(new T.bC("line without TAB"));t=(w.Rzb(0,P,w.Xzb(r).length),w.Xzb(r).substr(0,P)),e=T.Vhb(w.Xzb(T.Phb((w.Szb(P+1,w.Xzb(r).length+1),w.Xzb(r).substr(P+1))))),U.rj(this.a,new nt.rG(t,e));}},x(274,1,{},nt.qG),l.p9=X(274),nt.rG=function(t,e){this.a=t,this.b=e;},x(236,1,{236:1},nt.rG),y.b=0,l.o9=X(236),nt.sG=function(t,e){this.a=t,this.b=e;},x(45,1,{45:1},nt.sG),y.b=0,l.r9=X(45),nt.tG=function(t){t.a=new U.Lj();},nt.uG=function(t,e,A){U.rj(t.a,new nt.sG(e,A));},nt.vG=function(){nt.tG(this);},x(132,1,{},nt.vG),l.q9=X(132),nt.yG=function(){nt.yG=o1,nt.wG=F(k(l.Sbb,1),st,2,6,["[N](-*)(-*)-*","[N](-*)=*","[N]#*","[N](-*)(=*)=* as in nitro","[N](=*)#* middle atom of azide","[N]1(-*)-*-*-1 3-membered ring","[NH](-*)-*","[NH]1-*-*-1 3-membered ring","[NH]=*","[NH2]-*","[N+](-*)(-*)(-*)-*","[N+](-*)(-*)=*","[N+](-*)#* N in isocyano","[NH+](-*)(-*)-*","[NH+](-*)=*","[NH2+](-*)-*","[NH2+]=*","[NH3+]-*","[n](:*):*","[n](:*)(:*):*","[n](-*)(:*):*","[n](=*)(:*):* as in pyridine-N-oxid","[nH](:*):*","[n+](:*)(:*):*","[n+](-*)(:*):*","[nH+](:*):*","[O](-*)-*","[O]1-*-*-1 3-membered ring","[O]=*","[OH]-*","[O-]-*","[o](:*):*","[S](-*)-*","[S]=*","[S](-*)(-*)=*","[S](-*)(-*)(=*)=*","[SH]-*","[s](:*):*","[s](=*)(:*):*","[P](-*)(-*)-*","[P](-*)=*","[P](-*)(-*)(-*)=*","[PH](-*)(-*)=*"]),nt.xG=F(k(l.c7,1),w0,6,15,[3.240000009536743,12.359999656677246,23.790000915527344,11.680000305175781,13.600000381469727,hD,12.029999732971191,21.940000534057617,23.850000381469727,26.020000457763672,0,hD,4.360000133514404,4.440000057220459,13.970000267028809,16.610000610351562,25.59000015258789,27.639999389648438,12.890000343322754,4.409999847412109,4.929999828338623,8.390000343322754,15.789999961853027,4.099999904632568,3.880000114440918,14.140000343322754,9.229999542236328,12.529999732971191,17.06999969482422,20.229999542236328,23.059999465942383,13.140000343322754,25.299999237060547,32.09000015258789,19.209999084472656,8.380000114440918,38.79999923706055,28.239999771118164,21.700000762939453,13.59000015258789,34.13999938964844,9.8100004196167,23.469999313354492]);},nt.zG=function(t,e){var A,P;for(A=nt.DG(t),P=0;P<nt.xG.length;P++)A[P]!=0&&nt.uG(e,""+A[P]+" * "+nt.xG[P]+"   AtomType: "+nt.wG[P],2);},nt.AG=function(t){var e,A,P;for(e=nt.DG(t),P=0,A=0;A<nt.xG.length;A++)P+=e[A]*nt.xG[A];return P;},nt.BG=function(t){var e;return e=new nt.vG(),U.rj(e.a,new nt.sG("The polar surface area prediction is based on an atom-type based",2)),U.rj(e.a,new nt.sG("increment system, published by P. Ertl, B. Rohde, P. Selzer",2)),U.rj(e.a,new nt.sG("in J. Med. Chem. 2000, 43, 3714-3717",2)),U.rj(e.a,new nt.sG(Vr,2)),nt.zG(t,e),e;},nt.CG=function(t,e){var A;switch(t.C[e]){case 7:if(e<t.f&&n.nt(t.p,e)){if(t.s[e]==0){if(t.e[e]-n.sp(t,e)+n.np(t,e)==0){if(n.sp(t,e)==2)return 18;for(A=0;A<n.sp(t,e);A++)if(!n.Tp(t,t.k[e][A]))return 20;return 19;}else return 22;}else if(t.s[e]==1)if(t.e[e]-n.sp(t,e)+n.np(t,e)==0){for(A=0;A<n.sp(t,e);A++)if(!n.Tp(t,t.k[e][A]))return n.Al(t,t.i[e][A])<0?21:24;return 23;}else return 25;}else if(t.s[e]==0)switch(t.e[e]-n.sp(t,e)+n.np(t,e)){case 0:switch(t.o[e]){case 0:return (t.p&&e<t.f?n.ct(t.p,e):0)==3?5:0;case 1:return 1;case 2:return 2;}break;case 1:switch(t.o[e]){case 0:return (t.p&&e<t.f?n.ct(t.p,e):0)==3?7:6;case 1:return 8;}break;case 2:return 9;}else if(t.s[e]==1)switch(t.e[e]-n.sp(t,e)+n.np(t,e)){case 0:switch(t.o[e]){case 0:return 10;case 1:return nt.EG(t,e)?3:11;case 2:return t.n[e][0]==2?nt.EG(t,e)?4:nt.xG.length+1:12;}break;case 1:switch(t.o[e]){case 0:return 13;case 1:return 14;}break;case 2:return t.o[e]==0?15:16;case 3:return 17;}return nt.xG.length+1;case 8:if(e<t.f&&n.nt(t.p,e)){if(t.s[e]==0)return 31;}else {if(t.s[e]==0)return t.o[e]>0?28:n.sp(t,e)==1?29:(t.p&&e<t.f?n.ct(t.p,e):0)==3?27:26;if(t.s[e]==-1)return n.sp(t,e)==1&&n.Al(t,t.i[e][0])>0?28:30;}return nt.xG.length+1;case 15:if(t.s[e]==0){if(t.e[e]-n.sp(t,e)+n.np(t,e)==0){if(n.sp(t,e)==3&&t.o[e]==0)return 39;if(n.sp(t,e)==2&&t.o[e]==1)return 40;if(n.sp(t,e)==4&&t.o[e]==1)return 41;}else if(t.e[e]-n.sp(t,e)+n.np(t,e)==1&&n.sp(t,e)==3&&t.o[e]==1)return 42;}return nt.xG.length+1;case 16:if(t.s[e]==0){if(e<t.f&&n.nt(t.p,e))return n.sp(t,e)==2?37:38;if(t.e[e]-n.sp(t,e)+n.np(t,e)==0){if(n.sp(t,e)==2&&t.o[e]==0)return 32;if(n.sp(t,e)==1&&t.o[e]==1)return 33;if(n.sp(t,e)==3&&t.o[e]==1)return 34;if(n.sp(t,e)==4&&t.o[e]==2)return 35;}else if(t.e[e]-n.sp(t,e)+n.np(t,e)==1&&n.sp(t,e)==1)return 36;}return nt.xG.length+1;}return nt.xG.length;},nt.DG=function(t){var e,A;for(A=z(l.d7,H,6,nt.xG.length+2,15,1),n.aw(t,7),e=0;e<t.f;e++)++A[nt.CG(t,e)];return A;},nt.EG=function(t,e){var A;for(A=0;A<t.j[e];A++)if(n.Al(t,t.i[e][A])<0)return  true;return  false;},nt.FG=function(t){var e,A;for(A=0,e=0;e<t.a.q;e++)(n.Sl(t.a,e)==7||n.Sl(t.a,e)==8)&&++A;return A;},nt.GG=function(t){var e,A;for(A=0,e=0;e<t.a.q;e++)(n.Sl(t.a,e)==7||n.Sl(t.a,e)==8)&&n.Mo(t.a,e)>0&&++A;return A;},nt.HG=function(t){var e;try{return nt.fG((new nt.hG(),t.a));}catch(A){if(A=H1(A),Jt(A,19))return e=A,T.LB(e,(T.Ujb(),T.Tjb),""),-999;throw Ut(A);}},nt.IG=function(t){this.a=t,n.kq(t);},x(278,1,{}),l.s9=X(278),nt.LG=function(){nt.LG=o1,nt.JG=F(k(l.e7,1),n0,6,14,[524290,524292,524297,524301,589826,589827,589828,589829,589832,589833,589837,589838,688130,688131,688132,688137,688141,819202,819203,819204,819205,819208,819209,1114114,1114115,1114116,1114120,1212420,1343492,1343496,1638402,1638403,17367042,17367043,17367044,17367045,17367048,17367049,17367053,17367054,17465346,1074266116,1208483842,1208483844,1208549378,1208549379,1208549380,1208549384,1208549388,1409810434,1409875970,1409875971,1409875972,1409875976,1409974274,1409974280,1409974284,1678311426,1678311427,1678311428,1678311432,1678311436,1678409730,1678409731,1678409732,1678540802,1678540803,1678540804,1678540808,2282291202,2282291203,2282389506,2282389507,2282520578,2282520579,2483617794,2483617795,2483716098,2483716099,2752675842,3356033026,3557359618,3557457922,3557588994,34510798850,34510798851,34510798852,34510798856,34712125442,34712125443,34712125444,34712125448,34712223746,34712223747,34712223748,34980560898,34980560899,34980659202,35568287746,35568287747,35568287748,35568287752,35568386050,35568386051,35568386052,35568517122,35568517123,35568517124,35568517128,35568812034,35568812035,35568910338,35568910339,35568910350,35569336322,35585064962,35585064963,35585064964,35585064968,35769548802,35769614338,35769712642,35769843714,35786391554,ji,Bi,mi,ki,Fi,vi,pi,Ri,Ci,Hi,Mi,Si,qi,3437652377603,Vi,Ei,4674334294019,Ni,Ii,4674602827778,xi,Oi,Xi,Gi,5086651154435,Ki,5086651252739,Yi,5086651252744,Ji,5086919589896,Zi,5086919819266,_i,Wi,$i,5636675633154,{l:589826,m:73760,h:4},{l:589827,m:73760,h:4},{l:688130,m:73760,h:4},{l:819202,m:73760,h:4},{l:65538,m:73764,h:4},{l:65539,m:73764,h:4},{l:589826,m:172064,h:4},{l:589827,m:172064,h:4},{l:688130,m:172064,h:4},{l:819202,m:172064,h:4},{l:819203,m:172064,h:4},{l:65538,m:172068,h:4},{l:65539,m:172068,h:4},{l:589826,m:172112,h:4},{l:589827,m:172112,h:4},{l:688130,m:172112,h:4},{l:819202,m:172112,h:4},{l:65538,m:172116,h:4},{l:163842,m:172116,h:4},{l:589826,m:303136,h:4},{l:688130,m:303136,h:4},{l:819202,m:303136,h:4},{l:65538,m:303140,h:4},{l:589826,m:303184,h:4},{l:688130,m:303184,h:4},{l:819202,m:303184,h:4},{l:65538,m:303188,h:4},ai,tf,ef,nf,Af,Pf,rf,{l:688130,m:590224,h:4},{l:819202,m:590224,h:4},ff,Df,uf,of,lf,Qf,Lf,yf,{l:688135,m:590416,h:4},Uf,{l:1212430,m:590416,h:4},{l:65538,m:598052,h:4},{l:65538,m:598100,h:4},{l:65539,m:598100,h:4},{l:163842,m:598100,h:4},{l:163843,m:598100,h:4},sf,hf,Tf,{l:688131,m:598304,h:4},bf,cf,wf,gf,zf,{l:589826,m:688672,h:4},df,jf,Bf,mf,{l:589827,m:590112,h:336},kf,Ff,vf,pf,Rf,Cf,Hf,Mf,Sf,{l:688130,m:819536,h:400},qf,Vf,Ef,{l:589831,m:590112,h:592},Nf,If,xf,Of,Xf,Gf,{l:819207,m:819600,h:592},Kf,Yf,Jf,{l:589832,m:1212816,h:592},{l:688136,m:1212816,h:592},Zf,_f,Wf,$f,af,{l:589826,m:819488,h:8480},tD,eD,nD,AD,{l:819208,m:1213008,h:8480},PD,rD,{l:688130,m:590160,h:8484},{l:589826,m:598304,h:8484},iD,{l:819202,m:598304,h:8484}]),nt.KG=F(k(l.c7,1),w0,6,15,[-0.1899999976158142,1.2699999809265137,-0.7009999752044678,2.690999984741211,-0.22699999809265137,D9,.10599999874830246,-0.47600001096725464,-0.44699999690055847,-0.19099999964237213,-0.3330000042915344,.0860000029206276,.24699999392032623,-0.06199999898672104,.01600000075995922,.3869999945163727,.23499999940395355,-0.4320000112056732,-0.902999997138977,.38999998569488525,.5809999704360962,4.52400016784668,-0.6349999904632568,-0.2800000011920929,.7699999809265137,-0.05000000074505806,1.0870000123977661,.19200000166893005,.19599999487400055,-0.5199999809265137,.5419999957084656,.3630000054836273,.7919999957084656,.5920000076293945,.9639999866485596,oD,-0.6850000023841858,-0.3149999976158142,-0.4129999876022339,-0.5950000286102295,.2199999988079071,-1.4320000410079956,-2.253999948501587,.4399999976158142,-0.27000001072883606,-0.13300000131130219,-0.26899999380111694,.2669999897480011,.5720000267028809,-0.5680000185966492,.17399999499320984,-0.1850000023841858,-0.23499999940395355,TD,bD,-0.34200000762939453,-0.3479999899864197,-0.43700000643730164,-0.8040000200271606,-0.41200000047683716,-0.2150000035762787,-0.625,-0.8309999704360962,.4970000088214874,-0.4309999942779541,-1.3309999704360962,.5070000290870667,-0.6320000290870667,-0.5989999771118164,-0.15600000321865082,-0.3529999852180481,-0.164000004529953,-0.4410000145435333,-0.4970000088214874,-1.059999942779541,.11500000208616257,-0.22499999403953552,-0.15399999916553497,-0.03099999949336052,-1.5740000009536743,-1.093000054359436,yD,-0.44999998807907104,-0.5559999942779541,bD,2.384000062942505,1.75,-1.6660000085830688,-1.065999984741211,1.3270000219345093,.8029999732971191,-1.5049999952316284,-2.5369999408721924,TD,.14900000393390656,.5210000276565552,2.9049999713897705,-0.25200000405311584,.8600000143051147,.3610000014305115,.40299999713897705,.004999999888241291,1.1460000276565552,.9359999895095825,-0.30000001192092896,.20900000631809235,-0.5830000042915344,-0.024000000208616257,-0.009999999776482582,.6110000014305115,.4860000014305115,.8619999885559082,-0.03500000014901161,-0.5960000157356262,1.1610000133514404,1.6469999551773071,.843999981880188,.125,.1420000046491623,-0.17100000381469727,.44200000166893005,.08799999952316284,3.065999984741211,1.6519999504089355,cD,-0.017999999225139618,.023000000044703484,.0729999989271164,.2540000081062317,.5540000200271606,.5950000286102295,-0.4059999883174896,-0.6370000243186951,-0.17399999499320984,-0.10100000351667404,-0.5429999828338623,-2.4059998989105225,-3.2920000553131104,-0.05299999937415123,-0.19300000369548798,Hr,-1.2610000371932983,-0.656000018119812,-0.7300000190734863,-0.9380000233650208,.12800000607967377,1.1540000438690186,.24199999868869781,-0.5289999842643738,-0.27799999713897705,-0.8019999861717224,.9120000004768372,-1.38100004196167,.46299999952316284,1.0740000009536743,-0.628000020980835,-0.9620000123977661,-1.8320000171661377,-1.4989999532699585,-2.115999937057495,-2.2070000171661377,-1.3170000314712524,2.500999927520752,-0.8489999771118164,QD,-0.621999979019165,2.121999979019165,-2.2260000705718994,.9129999876022339,-2.259000062942505,-1.25,1.3940000534057617,-1.4019999504089355,2.072999954223633,-2.9570000171661377,.29100000858306885,-3.4760000705718994,-2.7269999980926514,-3.131999969482422,-2.119999885559082,-0.7250000238418579,-0.296999990940094,.08299999684095383,.34700000286102295,-1.4249999523162842,-1.659999966621399,-1.281999945640564,-1.2649999856948853,.718999981880188,.1379999965429306,1.3020000457763672,.859000027179718,1.3589999675750732,.6589999794960022,-0.9399999976158142,.8999999761581421,.3190000057220459,-2.571000099182129,1.1089999675750732,.972000002861023,1.652999997138977,2.6019999980926514,.7289999723434448,1.065999984741211,1.0670000314712524,-0.3109999895095825,.03099999949336052,cD,-0.6809999942779541,-1.2580000162124634,S9,-3.0959999561309814,-0.2280000001192093,1.9329999685287476,.11900000274181366,2.1080000400543213,.11299999803304672,1.628000020980835,1.3079999685287476,3.3359999656677246,.7540000081062317,-0.4650000035762787,-0.3970000147819519,.07699999958276749,-0.4790000021457672,-0.15299999713897705,.14100000262260437,2.134999990463257,.23399999737739563,.460999995470047,.6700000166893005,-0.3610000014305115,-1.0390000343322754,-0.4830000102519989,.13699999451637268,-0.7680000066757202,-0.5109999775886536,3.4240000247955322,-0.8550000190734863,-0.5849999785423279,-1.5670000314712524,3.3429999351501465,1.840000033378601,.3889999985694885,1.121999979019165,Mr,1.3350000381469727,.3659999966621399,-0.5569999814033508,.4320000112056732,.20399999618530273,.8820000290870667,.4659999907016754,-0.4580000042915344,.40400001406669617,.6570000052452087,1.1150000095367432,1.9759999513626099,1.7860000133514404,-0.035999998450279236,-1.0499999523162842,1.0449999570846558,.04399999976158142,1.0329999923706055,-1.0800000429153442,2.5390000343322754,2.234999895095825,Fr,3.121000051498413,3.931999921798706,2.75]);},nt.MG=function(t){var e,A,P,r;for(P=-0.5299999713897705,n.kq(t),n.aw(t,7),e=0;e<t.f;e++){r=-1;try{r=n.ef(t,e,2144);}catch(i){if(i=H1(i),!Jt(i,19))throw Ut(i);}for(A=0;A<nt.KG.length;A++)if(M1(nt.JG[A],r)){P+=nt.KG[A];break;}}return P;},nt.NG=function(t){var e,A,P,r,i,D,u;if(P=new nt.vG(),U.rj(P.a,new nt.sG("Solubility values are estimated applying an atom-type based increment system.",2)),U.rj(P.a,new nt.sG(UD,2)),U.rj(P.a,new nt.sG(Vr,2)),U.rj(P.a,new nt.sG("Base value = -0.530",2)),A=z(l.d7,H,6,nt.KG.length,15,1),t)for(n.kq(t),n.aw(t,7),e=0;e<t.f;e++){u=-1;try{u=n.ef(t,e,2144);}catch(o){if(o=H1(o),!Jt(o,19))throw Ut(o);}for(D=0;D<nt.KG.length;D++)if(M1(nt.JG[D],u)){++A[D];break;}}for(r=new o0.Mmb("#0.000"),i=0;i<nt.KG.length;i++)A[i]!=0&&nt.uG(P,""+A[i]+" * "+o0.Jmb(r,nt.KG[i])+"   AtomType: 0x"+T.Hib(nt.JG[i]),2);return P;},nt.UG=function(){nt.UG=o1,nt.OG=F(k(l.Sbb,1),st,2,6,["mutagenic","tumorigenic","irritant","reproductive effective"]),nt.PG=F(k(l.Sbb,1),st,2,6,["Mutagenicity","Tumorigenicity","Irritating effects","Reproductive effects"]);},nt.VG=function(t,e){var A,P,r,i;if(!nt.RG)return 0;if(nt.TG[e].ac(n.qg(new n.Mg(t)))!=-1)return 3;for(i=new n.iu(1),A=new n.sw(),r=0;r<nt.QG[e].a.length;r++)if(n.Sq(new n.$q(false),A,U.wj(nt.QG[e],r)),n.bu(i,t),n.$t(i,A),n.Ot(i,1,i.c)>0)return 3;for(P=0;P<nt.SG[e].a.length;P++)if(n.Sq(new n.$q(false),A,U.wj(nt.SG[e],P)),n.bu(i,t),n.$t(i,A),n.Ot(i,1,i.c)>0)return 2;return 1;},nt.WG=function(t,e){var A,P,r,i,D,u,o;if(o=new nt.vG(),!nt.RG)return U.rj(o.a,new nt.sG("Toxicity predictor not properly initialized.",2)),o;if(D=n.qg(new n.Mg(t)),nt.TG[e].ac(D)!=-1)return nt.uG(o,"This molecule is known to be "+nt.OG[e]+":",2),U.rj(o.a,new nt.sG(D,1)),o;for(u=new n.iu(1),A=false,P=new n.sw(),i=0;i<nt.QG[e].a.length;i++)n.Sq(new n.$q(false),P,U.wj(nt.QG[e],i)),n.bu(u,t),n.$t(u,P),n.Ot(u,1,u.c)>0&&(A||nt.uG(o,"High-risk fragments indicating "+nt.PG[e]+":",2),A=true,nt.uG(o,U.wj(nt.QG[e],i),1));for(A=false,r=0;r<nt.SG[e].a.length;r++)n.Sq(new n.$q(false),P,U.wj(nt.SG[e],r)),n.bu(u,t),n.$t(u,P),n.Ot(u,1,u.c)>0&&(A||nt.uG(o,"Medium-risk fragments indicating "+nt.PG[e]+":",2),A=true,nt.uG(o,U.wj(nt.SG[e],r),1));return o.a.a.length==0&&nt.uG(o,"No indication for "+nt.PG[e]+" found.",2),o;},nt.XG=function(t){var e,A,P,r;for(r=new Rt.igb(new Rt.ngb((A=_t.hAb(t),V1.Hmb(),A))),P=new n.Yv();;)try{if(e=Rt.hgb(r),e!=null)n.Wv(P,e);else break;}catch(i){if(i=H1(i),Jt(i,76))break;throw Ut(i);}return P;},nt.YG=function(t){var e,A,P,r;for(r=new Rt.igb(new Rt.ngb((P=_t.hAb(t),V1.Hmb(),P))),e=new U.Lj();;)try{if(A=Rt.hgb(r),A!=null)w.nzb(e.a,A);else break;}catch(i){if(i=H1(i),Jt(i,76))break;throw Ut(i);}return e;},nt.ZG=function(){if(nt.UG(),!nt.RG)try{nt.TG=z(l.c8,l1,102,4,0,1),nt.QG=z(l.zcb,l1,20,4,0,1),nt.SG=z(l.zcb,l1,20,4,0,1),nt.QG[0]=nt.YG("/resources/toxpredictor/m1.txt"),nt.QG[1]=nt.YG("/resources/toxpredictor/t1.txt"),nt.QG[2]=nt.YG("/resources/toxpredictor/i1.txt"),nt.QG[3]=nt.YG("/resources/toxpredictor/r1.txt"),nt.SG[0]=nt.YG("/resources/toxpredictor/m2.txt"),nt.SG[1]=nt.YG("/resources/toxpredictor/t2.txt"),nt.SG[2]=nt.YG("/resources/toxpredictor/i2.txt"),nt.SG[3]=nt.YG("/resources/toxpredictor/r2.txt"),nt.TG[0]=nt.XG("/resources/toxpredictor/m3.txt"),nt.TG[1]=nt.XG("/resources/toxpredictor/t3.txt"),nt.TG[2]=nt.XG("/resources/toxpredictor/i3.txt"),nt.TG[3]=nt.XG("/resources/toxpredictor/r3.txt"),nt.RG=!0;}catch(t){if(t=H1(t),Jt(t,19))T.Ujb();else throw Ut(t);}},x(284,1,{},nt.ZG),nt.RG=false,l.t9=X(284),tt.$G=function(t,e){U.rj(t.a,e);},tt._G=function(t,e,A){U.qj(t.a,A,e);},tt.aH=function(t,e){U.rj(t.f,e),t.d=-1;},tt.bH=function(t,e,A){U.qj(t.f,A,e),t.d=-1;},tt.cH=function(t,e){U.rj(t.g,e),t.d=-1;},tt.dH=function(t,e,A){U.qj(t.g,A,e),t.d=-1;},tt.eH=function(t){w.pzb(t.g.a,0),w.pzb(t.f.a,0),w.pzb(t.a.a,0),t.b=null,t.d=-1;},tt.fH=function(t){var e,A,P;for(P=new U.wqb(t.g);P.a<P.c.a.length;)if(e=U.vqb(P),e.K)return  true;for(A=new U.wqb(t.f);A.a<A.c.a.length;)if(e=U.vqb(A),e.K)return  true;return  false;},tt.gH=function(t){var e,A,P,r;for(A=0,e=0,P=0;P<t.g.a.length+t.f.a.length;P++)r=P<t.g.a.length?U.wj(t.g,P):U.wj(t.f,P-t.g.a.length),r.r!=0&&(A+=r.r,e+=n.Vl(r,r.q,r.r,(n.Qk(),n.Pk))*r.r);return A==0?(n.Qk(),n.Qk(),n.Pk):e/A;},tt.hH=function(t,e){return U.wj(t.a,e);},tt.iH=function(t){var e,A,P;if(t.d!=-1)return t.d;for(t.d=0,A=0;A<t.g.a.length+t.f.a.length;A++)for(P=A<t.g.a.length?U.wj(t.g,A):U.wj(t.f,A-t.g.a.length),e=0;e<P.q;e++)t.d<j.Math.abs(P.w[e])&&(t.d=j.Math.abs(P.w[e]));return t.d;},tt.jH=function(t){var e,A,P,r;if(A=new tt.yH(),t.g.a.length!=0){for(r=new n.uw(U.wj(t.g,0)),e=1;e<t.g.a.length;e++)n.Xk(r,U.wj(t.g,e));U.rj(A.g,r),A.d=-1;}if(t.f.a.length!=0){for(P=new n.uw(U.wj(t.f,0)),e=1;e<t.f.a.length;e++)n.Xk(P,U.wj(t.f,e));U.rj(A.f,P),A.d=-1;}return A;},tt.kH=function(t,e){return e<t.g.a.length?U.wj(t.g,e):U.wj(t.f,e-t.g.a.length);},tt.lH=function(t){return t.g.a.length+t.f.a.length;},tt.mH=function(t){return t.e==null?"":t.e;},tt.nH=function(t,e){return U.wj(t.f,e);},tt.oH=function(t,e){return U.wj(t.g,e);},tt.pH=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v;if(t.g.a.length==0||t.f.a.length==0||(h=tt.iH(t),h==0))return null;for(s=z(l.d7,S1,7,t.f.a.length,0,2),i=0;i<t.f.a.length;i++)for(g=U.wj(t.f,i),n.aw(g,15),s[i]=z(l.d7,H,6,h+1,15,1),U.Iqb(s[i],-1),e=0;e<g.q;e++){if(L=j.Math.abs(g.w[e]),L!=0&&s[i][L]!=-1)return null;s[i][L]=e;}for(D=z(l.zeb,ht,6,h+1,16,1),r=0;r<t.g.a.length;r++)for(v=U.wj(t.g,r),n.aw(v,15),m=0;m<v.q;m++)if(L=j.Math.abs(v.w[m]),L!=0&&!D[L]){for(u=0;u<t.f.a.length;u++)if(b=s[u][L],b!=-1){if(g=U.wj(t.f,u),v.j[m]!=g.j[b]){D[L]=true;break;}if((v.u[m]&3)!=(g.u[b]&3)){D[L]=true;break;}for(o=0;o<v.j[m];o++)if(A=n.Kl(v,v.i[m][o]),A==0)D[L]=true;else {for(C=v.k[m][o],P=false,Q=0;Q<g.j[b];Q++){if(d=n.Kl(g,g.i[b][Q]),d==0){D[L]=true;break;}if(d==A){if(P=true,c=g.k[b][Q],(C<v.g&&(n.tt(v.p,C)||v.H[C]==8))^(c<g.g&&(n.tt(g.p,c)||g.H[c]==8))||!(C<v.g&&(n.tt(v.p,C)||v.H[C]==8))&&(n.em(v,C)!=n.em(g,c)||(v.F[C]&3)!=(g.F[c]&3))){D[L]=true,D[A]=true;break;}break;}}P||(D[L]=true);}}}return D;},tt.qH=function(t){var e,A,P,r;for(P=new U.wqb(t.g);P.a<P.c.a.length;)if(e=U.vqb(P),e.q!=0)return  false;for(r=new U.wqb(t.f);r.a<r.c.a.length;)if(e=U.vqb(r),e.q!=0)return  false;for(A=new U.wqb(t.a);A.a<A.c.a.length;)if(e=U.vqb(A),e.q!=0)return  false;return  true;},tt.rH=function(t){return t.c||tt.fH(t);},tt.sH=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(A=0,s=new U.wqb(t.g);s.a<s.c.a.length;)if(Q=U.vqb(s),n.aw(Q,1),Q.K){for(e=0;e<Q.f;e++)if(kt(G(Q.B[e],r0),0)){if(j.Math.abs(Q.w[e])!=0)return  false;}else ++A;}else A+=Q.f;for(o=new U.wqb(t.f);o.a<o.c.a.length;)if(D=U.vqb(o),n.aw(D,1),D.K){for(e=0;e<D.f;e++)if(kt(G(D.B[e],r0),0)){if(j.Math.abs(D.w[e])!=0)return  false;}else --A;}else A-=D.f;if(A!=0)return  false;for(i=tt.iH(t),P=z(l.zeb,ht,6,i+1,16,1),L=new U.wqb(t.g);L.a<L.c.a.length;)for(Q=U.vqb(L),e=0;e<Q.f;e++)if(!Q.K||M1(G(Q.B[e],r0),0)){if(r=j.Math.abs(Q.w[e]),P[r])return  false;P[r]=true;}for(u=new U.wqb(t.f);u.a<u.c.a.length;)for(D=U.vqb(u),n.aw(D,1),e=0;e<D.f;e++)if(!D.K||M1(G(D.B[e],r0),0)){if(r=j.Math.abs(D.w[e]),r>=i||!P[r])return  false;P[r]=false;}return  true;},tt.tH=function(t){var e,A,P,r;if(t.g.a.length+t.f.a.length<=1)return  false;for(e=tt.gH(t),r=z(l.bab,C1,35,t.g.a.length+t.f.a.length,0,1),A=0;A<t.g.a.length+t.f.a.length;A++)if(r[A]=n.jm(A<t.g.a.length?U.wj(t.g,A):U.wj(t.f,A-t.g.a.length),null),r[A]){for(P=0;P<A;P++)if(r[P]&&(r[A].c+r[A].b>=r[P].c&&r[A].c<=r[P].c+r[P].b||r[A].d+r[A].a>=r[P].d&&r[A].d<=r[P].d+r[P].a))return  true;if(A!=0&&r[A-1]&&(r[A].c-r[A-1].c-r[A].b>5*e||r[A].d-r[A-1].d-r[A].a>5*e))return  true;}return  false;},tt.uH=function(t,e){t.b=e;},tt.vH=function(t,e){var A,P,r;for(t.c=e,r=new U.wqb(t.g);r.a<r.c.a.length;)A=U.vqb(r),n.Nn(A,e);for(P=new U.wqb(t.f);P.a<P.c.a.length;)A=U.vqb(P),n.Nn(A,e);},tt.wH=function(t,e){t.e=e;},tt.xH=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c;for(u=tt.iH(t),D=z(l.zeb,ht,6,u+1,16,1),c=new U.wqb(t.g);c.a<c.c.a.length;)for(h=U.vqb(c),A=0;A<h.q;A++)if(r=j.Math.abs(h.w[A]),r!=0){if(D[r])throw Ut(new T.bC("Duplicate mapping no in reactants"));D[r]=true;}for(i=z(l.zeb,ht,6,u+1,16,1),s=new U.wqb(t.f);s.a<s.c.a.length;)for(Q=U.vqb(s),A=0;A<Q.q;A++)if(r=j.Math.abs(Q.w[A]),r!=0){if(i[r])throw Ut(new T.bC("Duplicate mapping no in products"));i[r]=true;}for(o=z(l.d7,H,6,u+1,15,1),P=0,e=1;e<=u;e++)D[e]&&i[e]&&(o[e]=++P);if(P!=u){for(b=new U.wqb(t.g);b.a<b.c.a.length;)for(h=U.vqb(b),A=0;A<h.q;A++)n.qn(h,A,o[j.Math.abs(h.w[A])],h.w[A]<0);for(L=new U.wqb(t.f);L.a<L.c.a.length;)for(Q=U.vqb(L),A=0;A<Q.q;A++)n.qn(Q,A,o[j.Math.abs(Q.w[A])],Q.w[A]<0);}},tt.yH=function(){this.g=new U.Lj(),this.f=new U.Lj(),this.a=new U.Lj(),this.d=-1,this.c=false;},tt.zH=function(t){var e,A,P,r,i,D;for(tt.yH.call(this),D=t&&t.g?t.g.a.length:0,i=t&&t.f?t.f.a.length:0,e=t&&t.a?t.a.a.length:0,P=0;P<D;P++)U.rj(this.g,new n.uw(U.wj(t.g,P)));for(r=0;r<i;r++)U.rj(this.f,new n.uw(U.wj(t.f,r)));for(A=0;A<e;A++)U.rj(this.a,new n.uw(U.wj(t.a,A)));this.b=new n.fk(t.b),t.e!=null&&(this.e=t.e),this.c=t.c||tt.fH(t);},tt.AH=function(t,e){tt.yH.call(this),t!=null&&(U.tj(this.g,new U.qob(new U.irb(t),0,e)),U.tj(this.f,new U.qob(new U.irb(t),e,t.length))),this.c=tt.fH(this);},x(53,1,fn,tt.yH,tt.zH,tt.AH),y.c=false,y.d=0,l.v9=X(53),tt.BH=function(t,e,A){var P,r,i;return i=j.Math.sqrt((t.i[0].a-e)*(t.i[0].a-e)+(t.i[0].b-A)*(t.i[0].b-A)),i<8?1:(r=j.Math.sqrt((t.i[1].a-e)*(t.i[1].a-e)+(t.i[1].b-A)*(t.i[1].b-A)),r<8?2:(P=j.Math.sqrt((t.i[1].a-t.i[0].a)*(t.i[1].a-t.i[0].a)+(t.i[1].b-t.i[0].b)*(t.i[1].b-t.i[0].b)),i+r<P+5?3:0));},tt.CH=function(t){var e,A;return e=Ft(t.i[1].a-t.i[0].a),A=Ft(t.i[1].b-t.i[0].b),j.Math.sqrt(e*e+A*A);},tt.DH=function(t,e,A){var P,r,i,D,u,o;return P=t.i[1].a-t.i[0].a,r=t.i[1].b-t.i[0].b,D=(t.i[0].a+t.i[1].a)/2,u=(t.i[0].b+t.i[1].b)/2,P==0?r<0^A>u:r==0?P<0^e>D:(i=-P/r,o=(t.i[0].a+i*i*e-i*A+i*t.i[0].b)/(1+i*i),P<0^o>D);},tt.EH=function(t,e,A,P,r){t.i[0].a=e,t.i[0].b=A,t.i[1].a=P,t.i[1].b=r;},tt.FH=function(){this.i=z(l._9,C1,55,2,0,1),this.i[0]=new Tt.qN(),this.i[1]=new Tt.qN(),this.a=0;},tt.GH=function(t){var e,A,P,r;for(tt.FH.call(this),e=0;e!=-1&&(A=w.Xzb(t).indexOf('="',e),A!=-1);)if(P=(w.Rzb(e+1,A,w.Xzb(t).length),w.Xzb(t).substr(e+1,A-(e+1))),e=w.Xzb(t).indexOf('"',A+2),r=e==-1?(w.Szb(A+1,w.Xzb(t).length+1),w.Xzb(t).substr(A+1)):(w.Rzb(A+1,e,w.Xzb(t).length),w.Xzb(t).substr(A+1,e-(A+1))),T.ejb(P,"x1"))try{this.i[0].a=I0(r);}catch(i){if(i=H1(i),!Jt(i,40))throw Ut(i);}else if(T.ejb(P,"y1"))try{this.i[0].b=I0(r);}catch(i){if(i=H1(i),!Jt(i,40))throw Ut(i);}else if(T.ejb(P,"x2"))try{this.i[1].a=I0(r);}catch(i){if(i=H1(i),!Jt(i,40))throw Ut(i);}else if(T.ejb(P,"y2"))try{this.i[1].b=I0(r);}catch(i){if(i=H1(i),!Jt(i,40))throw Ut(i);}},x(61,193,{193:1,61:1},tt.FH,tt.GH),y.Db=function(t,e){return this.a=tt.BH(this,t,e),this.a!=0;},y.Eb=function(){this.a=0;},y.Fb=function(){var t;return t=new tt.FH(),t.i[0].a=this.i[0].a,t.i[0].b=this.i[0].b,t.i[1].a=this.i[1].a,t.i[1].b=this.i[1].b,t.g=this.g,t;},y.Gb=function(t,e){return tt.BH(this,t,e)!=0;},y.Hb=function(t,e){var A,P,r,i,D,u,o;R.V2(t,this.g?lt.N3(W0,R.M2(t)):R.P2(t)),i=e?n.zi(e,this.i[0].a):this.i[0].a,u=e?n.Ai(e,this.i[0].b):this.i[0].b,D=e?n.zi(e,this.i[1].a):this.i[1].a,o=e?n.Ai(e,this.i[1].b):this.i[1].b,A=D-i,P=o-u,R.U2(t,j.Math.max(1,.019999999552965164*j.Math.sqrt(A*A+P*P))),R.F2(t,i,u,D,o),r=new Tt.yN(4),Tt.tN(r,D+A/40,o+P/40),Tt.tN(r,D-A/5+P/10,o-P/5-A/10),Tt.tN(r,D-A/20,o-P/20),Tt.tN(r,D-A/5-P/10,o-P/5+A/10),R.K2(t,new R.i3(r));},y.Ib=function(t){var e,A,P,r,i,D;return D=tt.CH(this),i=j.Math.max(D/8,3),e=n.mo(this.i[0].a,this.i[0].b,this.i[1].a,this.i[1].b),P=j.Math.abs(i*j.Math.cos(e)),r=j.Math.abs(i*j.Math.sin(e)),A=new Tt.EN(),this.i[0].a<this.i[1].a?(A.c=this.i[0].a-P,A.b=this.i[1].a-this.i[0].a+2*P):(A.c=this.i[1].a-P,A.b=this.i[0].a-this.i[1].a+2*P),this.i[0].b<this.i[1].b?(A.d=this.i[0].b-r,A.a=this.i[1].b-this.i[0].b+2*r):(A.d=this.i[1].b-r,A.a=this.i[0].b-this.i[1].b+2*r),A;},y.Jb=function(){var t;return t=new T.Pjb(),T.Njb(t,' x1="'+this.i[0].a+'"'),T.Njb(t,' y1="'+this.i[0].b+'"'),T.Njb(t,' x2="'+this.i[1].a+'"'),T.Njb(t,' y2="'+this.i[1].b+'"'),t.a;},y.Kb=function(){return "arrow";},y.Lb=function(t){var e,A,P,r,i,D;switch(R.V2(t,R.R2(t)),this.a){case 1:R.J2(t,this.i[0].a-8,this.i[0].b-8,16);break;case 2:R.J2(t,this.i[1].a-8,this.i[1].b-8,16);break;case 3:i=tt.CH(this),r=j.Math.max(i/8,3),e=n.mo(this.i[0].a,this.i[0].b,this.i[1].a,this.i[1].b),A=r*j.Math.cos(e),P=-r*j.Math.sin(e),D=new Tt.yN(4),Tt.tN(D,this.i[0].a+A,this.i[0].b+P),Tt.tN(D,this.i[1].a+A,this.i[1].b+P),Tt.tN(D,this.i[1].a-A,this.i[1].b-P),Tt.tN(D,this.i[0].a-A,this.i[0].b-P),R.K2(t,new R.i3(D));}},y.Nb=function(t,e){switch(this.a){case 1:this.i[0].a=this.o[0]+t-this.k,this.i[0].b=this.p[0]+e-this.n;break;case 2:this.i[1].a=this.o[1]+t-this.k,this.i[1].b=this.p[1]+e-this.n;break;default:n.ne(this,t,e);}},y.a=0,l.u9=X(61),tt.RH=function(t,e){var A,P,r,i,D,u,o,Q,L;return t==null?null:(u=t,Q=null,o=null,L=null,D=null,A=T.ljb(t,Z1(35)),A==-1?u=t:(u=(w.Rzb(0,A,w.Xzb(t).length),w.Xzb(t).substr(0,A)),P=T.mjb(t,Z1(35),A+1),P==-1?Q=(w.Szb(A+1,w.Xzb(t).length+1),w.Xzb(t).substr(A+1)):(Q=(w.Rzb(A+1,P,w.Xzb(t).length),w.Xzb(t).substr(A+1,P-(A+1))),r=T.mjb(t,Z1(35),P+1),r==-1?o=(w.Szb(P+1,w.Xzb(t).length+1),w.Xzb(t).substr(P+1)):(o=(w.Rzb(P+1,r,w.Xzb(t).length),w.Xzb(t).substr(P+1,r-(P+1))),i=T.mjb(t,Z1(35),r+1),i==-1?L=(w.Szb(r+1,w.Xzb(t).length+1),w.Xzb(t).substr(r+1)):(L=(w.Rzb(r+1,i,w.Xzb(t).length),w.Xzb(t).substr(r+1,i-(r+1))),D=(w.Szb(i+1,w.Xzb(t).length+1),w.Xzb(t).substr(i+1)))))),tt.SH(u,(e&1)!=0?Q:null,(e&2)!=0?o:null,(e&4)!=0?L:null,(e&8)!=0?D:null,false,null));},tt.SH=function(t,e,A,P,r,i,D){var u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V;if(t==null||w.Xzb(t).length==0||(c=false,s=0,v=0,o=0,V=T.ljb(t,Z1(33)),V==-1))return null;for(D?tt.eH(D):D=new tt.yH();s!=-1;)s>V&&(c=true),Q=T.mjb(t,Z1(32),s),!c&&(Q>V||Q==-1)&&(Q=V),L=null,Q==-1?(L=(w.Szb(s,w.Xzb(t).length+1),w.Xzb(t).substr(s)),s=-1):(L=(w.Rzb(s,Q,w.Xzb(t).length),w.Xzb(t).substr(s,Q-s)),s=Q+1),C=null,e!=null&&w.Xzb(e).length!=0&&(Q=T.mjb(e,Z1(32),v),Q==-1?C=(w.Szb(v,w.Xzb(e).length+1),w.Xzb(e).substr(v)):(C=(w.Rzb(v,Q,w.Xzb(e).length),w.Xzb(e).substr(v,Q-v)),v=Q+1)),u=null,A!=null&&w.Xzb(A).length!=0&&(Q=T.mjb(A,Z1(32),o),Q==-1?u=(w.Szb(o,w.Xzb(A).length+1),w.Xzb(A).substr(o)):(u=(w.Rzb(o,Q,w.Xzb(A).length),w.Xzb(A).substr(o,Q-o)),o=Q+1)),S=new n.$q(i),M=L==null?null:n.Rq(S,w.xzb((d=L,V1.Hmb(),d)),u==null?null:w.xzb((m=u,m)),0),C!=null&&n.Vq(S,w.xzb((g=C,V1.Hmb(),g))),c?(U.rj(D.f,M),D.d=-1):(U.rj(D.g,M),D.d=-1);if(P!=null&&w.Xzb(P).length!=0&&tt.uH(D,new n.gk(P)),r!=null&&w.Xzb(r).length!=0){for(S=new n.$q(i),h=0,b=T.ljb(r,Z1(43));b!=-1;)tt.$G(D,n.Oq(S,(w.Rzb(h,b,w.Xzb(r).length),w.Xzb(r).substr(h,b-h)))),h=b+1,b=T.mjb(r,Z1(43),h);tt.$G(D,n.Oq(S,(w.Szb(h,w.Xzb(r).length+1),w.Xzb(r).substr(h))));}return D;},tt.TH=function(t,e,A){var P,r,i,D,u,o,Q,L,s;return t==null?null:(o=t,L=null,Q=null,s=null,u=null,P=T.ljb(t,Z1(35)),P==-1?o=t:(o=(w.Rzb(0,P,w.Xzb(t).length),w.Xzb(t).substr(0,P)),r=T.mjb(t,Z1(35),P+1),r==-1?L=(w.Szb(P+1,w.Xzb(t).length+1),w.Xzb(t).substr(P+1)):(L=(w.Rzb(P+1,r,w.Xzb(t).length),w.Xzb(t).substr(P+1,r-(P+1))),i=T.mjb(t,Z1(35),r+1),i==-1?Q=(w.Szb(r+1,w.Xzb(t).length+1),w.Xzb(t).substr(r+1)):(Q=(w.Rzb(r+1,i,w.Xzb(t).length),w.Xzb(t).substr(r+1,i-(r+1))),D=T.mjb(t,Z1(35),i+1),D==-1?s=(w.Szb(i+1,w.Xzb(t).length+1),w.Xzb(t).substr(i+1)):(s=(w.Rzb(i+1,D,w.Xzb(t).length),w.Xzb(t).substr(i+1,D-(i+1))),u=(w.Szb(D+1,w.Xzb(t).length+1),w.Xzb(t).substr(D+1)))))),tt.SH(o,L,Q,s,u,e,A));},tt.UH=function(t,e,A){var P,r;return r=tt.VH(t,e,(A&16)==0),r==null?null:(P=new T.Ijb(r[0]),A!=0&&(P.a+="#",(A&1)!=0&&r.length>1&&r[1]!=null&&T.Fjb(P,r[1])),A&=-2,A!=0&&(P.a+="#",(A&2)!=0&&r.length>2&&r[2]!=null&&T.Fjb(P,r[2])),A&=-3,A!=0&&(P.a+="#",(A&4)!=0&&r.length>3&&r[3]!=null&&T.Fjb(P,r[3])),A&=-5,A!=0&&(P.a+="#",(A&8)!=0&&r.length>4&&r[4]!=null&&T.Fjb(P,r[4])),P.a);},tt.VH=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;if(!t||t.g.a.length==0||t.f.a.length==0)return null;for(L=z(l.Sbb,st,2,t.g.a.length+t.f.a.length,6,1),c=z(l.Sbb,st,2,t.g.a.length+t.f.a.length,6,1),i=z(l.Sbb,st,2,t.g.a.length+t.f.a.length,6,1),o=0;o<t.g.a.length+t.f.a.length;o++){if(m=o<t.g.a.length?U.wj(t.g,o):U.wj(t.f,o-t.g.a.length),m.K&&o<t.g.a.length)for(P=0;P<m.q;P++)n.un(m,P,he,false);if(r=new n.Mg(m),L[o]=n.qg(r),L[o]==null)return null;c[o]=n.ng(r),i[o]=n.mg(r,e);}for(s=new T.Pjb(),D=new T.Pjb(),g=new T.Pjb(),Q=0;Q<t.g.a.length;Q++){if(h=Q,A)for(d="",h=-1,b=0;b<t.g.a.length;b++)T.$ib(d,L[b])<0&&(d=L[b],h=b);Q>0&&(s.a+=" ",g.a+=" ",D.a+=" "),T.Njb(s,L[h]),T.Njb(g,c[h]),T.Njb(D,i[h]),L[h]="";}for(s.a+="!",g.a+=" ",D.a+=" ",u=t.g.a.length;u<t.g.a.length+t.f.a.length;u++){if(h=u,A)for(d="",h=-1,b=t.g.a.length;b<t.g.a.length+t.f.a.length;b++)T.$ib(d,L[b])<0&&(d=L[b],h=b);u>t.g.a.length&&(s.a+=" ",g.a+=" ",D.a+=" "),T.Njb(s,L[h]),T.Njb(g,c[h]),T.Njb(D,i[h]),L[h]="";}return C=z(l.Sbb,st,2,5,6,1),C[0]=s.a,w.Xzb(g.a).length>t.g.a.length+t.f.a.length-1&&(C[1]=g.a),w.Xzb(D.a).length>t.g.a.length+t.f.a.length-1&&(C[2]=D.a),t.b&&(C[3]=n.dk(t.b)),t.a.a.length!=0&&(C[4]=tt.WH(t,e)),C;},tt.WH=function(t,e){var A,P,r;for(r=new T.Pjb(),P=0;P<t.a.a.length;P++)w.Xzb(r.a).length!=0&&(r.a+="+"),A=new n.Mg(U.wj(t.a,P)),T.Njb(r,n.qg(A)),e&&(r.a+=" ",T.Njb(r,n.mg(A,true)));return r.a;},tt.XH=function(t,e,A,P,r){var i,D,u,o;if((e.u[A]&3)!=0&&(e.u[A]&4)==0&&M1(G(e.B[A],he),0)&&(u=j.Math.abs(e.w[A]),u!=0)){for(D=0;D<t.c.g.a.length;D++)for(i=tt.oH(t.c,D),o=0;o<i.f;o++)if(j.Math.abs(i.w[o])==u)return !!(e.j[A]==P.j[r]||kt(G(i.B[o],Q0),0)||(i.u[o]&U1)>>18!=(e.u[A]&U1)>>18);}return  false;},tt.YH=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b;for(h=z(l.d7,H,6,t.f,15,1),b=tt._H(t,e,h),D=false,i=false,s=new U.Lj(),r=0;r<b;r++)n.xm(t,h[r])&&(Q=n.pw(t,h[r],P),(Q==1||Q==2)&&(u=n.Ml(A,P[h[r]]),(u==1||u==2)&&(Q==u?D=true:i=true)),o=z(l.d7,H,6,2,15,1),o[0]=h[r],o[1]=Q,w.nzb(s.a,o));if(!D||!i)for(L=new U.wqb(s);L.a<L.c.a.length;)Q=U.vqb(L),o=Q[1],(o==1||o==2)&&i&&(o=o==1?2:1),n.tn(A,P[Q[0]],o,true);},tt.ZH=function(t){var e,A,P,r,i,D,u,o,Q,L,s;if(t.n==null){if(t.b)for(P=t.j+1,t.p=z(l.d7,c0,28,t.o.length,0,3),i=0;i<t.o.length;i++){for(t.p[i]=z(l.d7,S1,7,t.g[i].a.length,0,2),u=0;u<t.g[i].a.length;u++)t.p[i][u]=tt.dI(t,i,U.wj(t.g[i],u),P);for(P+=t.o[i].f,r=tt.oH(t.c,i),e=0;e<r.f;e++)kt(G(r.B[e],r0),0)||--P;}if(t.n=h1(l.d8,[c0,xe],[75,25],0,[t.q,t.c.f.a.length],2),t.d=h1(l.Sbb,[c0,st],[13,2],6,[t.q,t.c.f.a.length],2),t.e=z(l.zeb,ht,6,t.q,16,1),s=new n.Yv(),L=new T.Pjb(),D=0,A=0,t.q!=0){o=z(l.d7,H,6,t.o.length,15,1);do{for(Q=w.Xzb(L.a).length,0<Q?L.a=T.vjb(L.a,0,0):0>Q&&(L.a+=T.Bjb(z(l.a7,n0,6,-Q,15,1))),i=0;i<t.c.f.a.length;i++)a1(t.n[D],i,tt.aI(t,t.g,o,i)),t.d[D][i]=n.qg(new n.Mg(t.n[D][i])),T.Njb(L,t.d[D][i]);t.e[D]=n.Wv(s,L.a)==-1,(!t.t||!t.e[D])&&++A,++D;}while(tt.gI(t,o)&&A<t.i);}}},tt._H=function(t,e,A){var P,r,i,D,u,o;for(u=z(l.zeb,ht,6,t.f,16,1),A[0]=e,o=1,u[e]=true,i=0;i<o;i++)for(D=0;D<n.Zo(t,A[i]);D++)r=n.$o(t,A[i],D),((t.F[r]&64)!=0||n.em(t,r)==2||n.Up(t,r))&&(P=n.Yo(t,A[i],D),u[P]||(A[o++]=P,u[P]=true));return o;},tt.aI=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct,wt,$t,vt,Ht,xt,Ot,Nt,pt,mt,y1,u1,Mt,Zt,m1,Et,f1,x1,s1,Q1,I1,G1,v0,K1,L1,D0,T0,j0,pe,ge,ne,Ae,R0,Re,K0,C0;for(K=tt.nH(t.c,P),L1=new n.sw(),g=0,d=0,K0=null,it=0;it<t.o.length;it++){for(N=tt.oH(t.c,it),n.aw(t.o[it],1),s1=U.wj(e[it],A[it]),x1=z(l.d7,H,6,t.o[it].f,15,1),v=z(l.zeb,ht,6,t.o[it].f,16,1),M=z(l.zeb,ht,6,t.o[it].g,16,1),Ot=0;Ot<N.f;Ot++)if(s1[Ot]!=-1)if(j.Math.abs(N.w[Ot])==0){for(S=s1[Ot],v[S]=true,m1=0;m1<n.Zo(t.o[it],S);m1++)M[n.$o(t.o[it],S,m1)]=true;for(Zt=n.Ko(t.o[it],S);Zt<n.Lo(t.o[it],S);Zt++)M[n.$o(t.o[it],S,Zt)]=true;}else x1[s1[Ot]]=j.Math.abs(N.w[Ot]);for(Nt=0;Nt<N.g;Nt++)if(s=N.D[0][Nt],h=N.D[1][Nt],j.Math.abs(N.w[s])!=0&&j.Math.abs(N.w[h])!=0&&(u=s1[s],o=s1[h],u!=-1&&o!=-1)){for(Zt=0;Zt<t.o[it].g;Zt++)if(n.Yl(t.o[it],0,Zt)==u&&n.Yl(t.o[it],1,Zt)==o||n.Yl(t.o[it],0,Zt)==o&&n.Yl(t.o[it],1,Zt)==u){M[Zt]=true;break;}}for(G1=z(l.d7,H,6,t.o[it].f,15,1),pt=0;pt<t.o[it].f;pt++)if(!v[pt]&&(G1[pt]=n.kl(t.o[it],L1,pt,g,d),t.b&&n.qn(L1,G1[pt],t.p[it][A[it]][pt],false),x1[pt]!=0)){for(n.qn(L1,G1[pt],x1[pt],false),Zt=0;Zt<K.f;Zt++)if(j.Math.abs(K.w[Zt])==x1[pt]){if(n.hn(L1,G1[pt],K.s[Zt]),n.vn(L1,G1[pt],K.u[Zt]&48),t.f[P][Zt])if(K1=n.Ml(t.o[it],pt),T0=G(K.B[Zt],he),et(T0,0)==0)n.tn(L1,G1[pt],3,false);else if(et(T0,he)==0)!K0&&(K0=new U.Lj()),U.rj(K0,T.uib(G1[pt]));else if(K1==1||K1==2){for(wt=et(T0,dA)==0,Et=0;Et<n.Zo(t.o[it],pt);Et++)if(b=n.Yo(t.o[it],pt,Et),v[b])for(f1=0;f1<n.Zo(t.o[it],pt);f1++)v0=n.Yo(t.o[it],pt,f1),!v[v0]&&v0>b&&(wt=!wt);wt&&(K1=K1==1?2:1),n.tn(L1,G1[pt],K1,false);}else n.tn(L1,G1[pt],0,false);break;}}for(mt=0;mt<t.o[it].g;mt++)M[mt]||n.ml(t.o[it],L1,mt,g,d,G1,true);g=n.bn(L1,1),d=n.bn(L1,2);}for(I1=z(l.d7,H,6,K.f,15,1),y1=0;y1<K.f;y1++)if(x1=j.Math.abs(K.w[y1]),x1==0)I1[y1]=n.kl(K,L1,y1,g,d);else for(Zt=0;Zt<L1.q;Zt++)if(j.Math.abs(L1.w[Zt])==x1){I1[y1]=Zt;break;}if(t.r)for(Ot=0;Ot<K.f;Ot++)n.rn(L1,I1[Ot],true),n.yn(L1,I1[Ot],K.J[Ot].a),n.zn(L1,I1[Ot],K.J[Ot].b);for(I=0;I<K.g;I++)if((K.G[I]&Le)==0&&(D0=n.ll(K,L1,I,g,d,I1[K.D[0][I]],I1[K.D[1][I]],false),j0=K.G[I]&31,j0!=0))for(O=false,n.Kn(L1,D0,31,false),ut=n.Kl(K,K.D[0][I]),Qt=n.Kl(K,K.D[1][I]),bt=0;!O&&bt<t.c.g.a.length;bt++)for(N=tt.oH(t.c,bt),ot=0;!O&&ot<N.g;ot++)Dt=N.D[0][ot],Lt=N.D[1][ot],rt=j.Math.abs(N.w[Dt]),yt=j.Math.abs(N.w[Lt]),rt!=0&&yt!=0&&(rt==ut&&yt==Qt||rt==Qt&&yt==ut)&&(O=true,C0=N.G[ot]&31,C0!=0&&(s1=U.wj(e[bt],A[bt]),ge=s1[Dt],ne=s1[Lt],Ae=n.Wo(t.o[bt],ge,ne),Re=n.hm(t.o[bt],Ae),R0=n.em(t.o[bt],Ae),C0&=7,j0&=7,C0==3&&j0==6?(L1.H[D0]=R0<=1?2:4,L1.T=0):C0==6&&j0==3?(L1.H[D0]=R0==3?2:1,L1.T=0):(L1.H[D0]=Re,L1.T=0)));for(n.aw(L1,1),m=false,u1=0;u1<K.f;u1++)j.Math.abs(K.w[u1])==0?(K1=n.pw(K,u1,I1),n.tn(L1,I1[u1],K1,false)):t.f[P][u1]&&tt.XH(t,K,u1,L1,I1[u1])&&(K1=n.pw(K,u1,I1),n.tn(L1,I1[u1],K1,false),(K1==1||K1==2)&&(C=(K.u[u1]&U1)>>18,c=(K.u[u1]&U1)>>18!=1&&(K.u[u1]&U1)>>18!=2?-1:(K.u[u1]&k0)>>20,C==1?c+=g:C==2&&(c+=d),n.nn(L1,I1[u1],C,c),m=true));for(m&&(g=n.bn(L1,1),d=n.bn(L1,2)),pe=z(l.zeb,ht,6,K.q,16,1),Mt=0;Mt<K.f;Mt++)t.f[P][Mt]&&!pe[Mt]&&(K.u[Mt]&4)!=0&&M1(G(K.B[Mt],he),0)&&K.j[Mt]==n.Zo(L1,I1[Mt])&&tt.YH(K,Mt,L1,I1);for(ct=z(l.zeb,ht,6,L1.q,16,1),vt=I1,Ht=0,xt=vt.length;Ht<xt;++Ht)$t=vt[Ht],ct[$t]=true;for(V=true;V;)for(V=false,L=0;L<L1.r;L++)u=L1.D[0][L],o=L1.D[1][L],ct[u]&&!ct[o]?(ct[o]=true,V=true):ct[o]&&!ct[u]&&(ct[u]=true,V=true);for(D=0;D<L1.q;D++)n.wn(L1,D,!ct[D]);if(n.vl(L1),!t.b)for(Q=0;Q<L1.q;Q++)L1.w[Q]=0;if(n.Fe(new n.Ue(L1),null,false),L1.T|=8,Q1=2|(t.r?8:0),Z.iA(new Z.BA(Q1),L1),K0)for(n.aw(L1,15),i=new U.wqb(K0);i.a<i.c.a.length;)r=U.vqb(i).a,(L1.u[r]&q0)!=0&&(n.tn(L1,r,1,false),n.nn(L1,r,1,g++));return L1;},tt.bI=function(t){var e,A;if(e=t.q,t.t)for(A=0;A<t.q;A++)t.e[A]&&--e;return t.i>0&&e>t.i&&(e=t.i),e;},tt.cI=function(t){var e,A,P;for(tt.ZH(t),P=z(l.d8,c0,75,tt.bI(t),0,2),e=0,A=0;A<t.q&&e<t.i;A++)(!t.t||!t.e[A])&&(P[e++]=t.n[A]);return P;},tt.dI=function(t,e,A,P){var r,i,D,u;for(u=z(l.d7,H,6,t.o[e].f,15,1),U.Oqb(u,u.length,-1),i=tt.oH(t.c,e),r=0;r<i.f;r++)A[r]!=-1&&(u[A[r]]=j.Math.abs(i.w[r]));for(D=0;D<u.length;D++)u[D]==-1&&(u[D]=P++);return u;},tt.eI=function(t,e){var A,P,r,i;for(A=z(l.d7,H,6,t.j[e]-n.ap(t,e),15,1),i=0,r=0;r<t.j[e];r++)P=t.i[e][r],kt(G(t.B[P],r0),0)||(A[i++]=t.C[P]);return w.qzb(A,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),A;},tt.fI=function(t,e){var A,P,r,i;for(i=z(l.d7,H,6,t.j[e]-n.ap(t,e),15,1),r=0,P=0;P<t.j[e];P++)A=t.i[e][P],kt(G(t.B[A],r0),0)||(i[r++]=j.Math.abs(t.w[A]));return w.qzb(i,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),i;},tt.gI=function(t,e){var A;for(A=0;A<e.length;A++){if(e[A]<t.g[A].a.length-1)return ++e[A],true;e[A]=0;}return  false;},tt.hI=function(t,e){var A,P,r,i,D,u,o,Q,L,s;for(t.j=0,r=0;r<e.g.a.length+e.f.a.length;r++)for(o=tt.kH(t.c,r),A=0;A<o.f;A++)kt(G(o.B[A],r0),0)?o.w[A]=0:t.j<j.Math.abs(o.w[A])&&(t.j=j.Math.abs(o.w[A]));for(s=z(l.zeb,ht,6,t.j+1,16,1),i=0;i<e.g.a.length;i++)for(L=U.wj(e.g,i),A=0;A<L.f;A++)u=j.Math.abs(L.w[A]),u!=0&&(s[u]=true);for(D=0;D<e.f.a.length;D++)for(Q=U.wj(e.f,D),A=0;A<Q.f;A++)u=j.Math.abs(Q.w[A]),u!=0&&(s[u]?s[u]=false:Q.w[A]=0);for(P=0;P<e.g.a.length;P++)for(L=U.wj(e.g,P),A=0;A<L.f;A++)u=j.Math.abs(L.w[A]),u!=0&&s[u]&&(L.w[A]=0);},tt.iI=function(t,e,A){var P,r,i,D,u,o,Q,L;if(t.o[e]=A,t.n=null,r=tt.oH(t.c,e),n.au(t.s,r,t.o[e]),o=8+(t.a?0:1),n.Ot(t.s,5,o)==0)return t.g[e]=new U.Lj(),t.q=0,false;for(t.g[e]=t.s.F,D=t.g[e].a.length-1;D>=0;D--)for(Q=U.wj(t.g[e],D),u=0;u<Q.length;u++)if(Q[u]!=-1&&(P=0,t.a&&r.s[u]==0&&n.Al(t.o[e],Q[u])!=0&&(L=n.om(t.o[e],Q[u]),P=n.lm(t.o[e],Q[u],L)),t.k[e][u]>0&&t.k[e][u]>n.ip(t.o[e],Q[u])-P)){U.Dj(t.g[e],D);break;}for(t.q=1,i=0;i<t.o.length;i++)t.q*=t.g[i]?t.g[i].a.length:0;return  true;},tt.jI=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b;for(this.r=true,this.b=false,this.t=true,this.a=false,this.i=St,this.c=t,this.o=z(l.d8,xe,25,t.g.a.length,0,1),P=0;P<t.g.a.length;P++)n.Nn(U.wj(t.g,P),true),n.aw(U.wj(t.g,P),15);for(r=0;r<t.f.a.length;r++)n.aw(U.wj(t.f,r),15);for(tt.hI(this,t),this.k=z(l.d7,S1,7,t.g.a.length,0,2),i=0;i<t.g.a.length;i++)for(s=U.wj(t.g,i),this.k[i]=z(l.d7,H,6,s.f,15,1),D=0;D<s.f;D++)if(Q=j.Math.abs(s.w[D]),Q!=0)for(u=0;u<t.f.a.length;u++)for(L=U.wj(t.f,u),o=0;o<L.f;o++)j.Math.abs(L.w[o])==Q&&(e=(b=n.om(s,D),b+n.mm(s,D,b,s.C[D])-n.up(s,D)-(h=n.om(L,o),h+n.mm(L,o,h,L.C[o])-n.up(L,o))),this.k[i][D]=j.Math.max(e,0));for(this.f=z(l.zeb,l1,12,t.f.a.length,0,2),A=0;A<t.f.a.length;A++)for(L=U.wj(t.f,A),this.f[A]=z(l.zeb,ht,6,L.f,16,1),D=0;D<L.f;D++)if(Q=j.Math.abs(L.w[D]),Q!=0)for(u=0;u<t.g.a.length;u++)for(s=U.wj(t.g,u),o=0;o<s.f;o++)j.Math.abs(s.w[o])==Q&&(lt.U3(tt.fI(L,D),tt.fI(s,o))!=0||lt.U3(tt.eI(L,D),tt.eI(s,o))!=0)&&(this.f[A][D]=true);this.s=new n.hu(),this.g=z(l.zcb,l1,20,this.o.length,0,1);},tt.kI=function(t){tt.jI.call(this,t);},x(214,1,{},tt.kI),y.a=false,y.b=false,y.i=0,y.j=0,y.q=0,y.r=false,y.t=false,l.w9=X(214),tt.lI=function(t,e,A){var P,r,i;for(n.aw(e,1),r=U.wj(t.a,A),P=0;P<t.d.d.a.length;P++)i=lt._3(t.d,P),tt.oI(i,e,r);n.ul(e);},tt.mI=function(t,e){var A,P,r;if(t.f=e,n.au(t.e,t.c,t.f),n.Ot(t.e,4,8)==0)return t.a=null,0;for(t.a=t.e.F,A=t.a.a.length-1;A>=0;A--)for(r=U.wj(t.a,A),P=0;P<r.length;P++)if(r[P]!=-1&&t.b[P]>0&&t.b[P]>n.ip(t.f,r[P])){U.Dj(t.a,A);break;}return t.a.a.length;},tt.nI=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;for(n.aw(t,7),n.aw(e,7),this.c=t,this.d=new lt.d4(),P=0;P<t.g;P++)lt.Z3(this.d,new tt.rI(t,P));for(A=0;A<e.g;A++)L=new tt.rI(e,A),o=(u=lt.a4(this.d,L),u<0?-1:u),o==-1?(L.i=(tt.AI(),tt.wI),lt.Z3(this.d,L)):tt.qI(lt._3(this.d,o),L);for(D=this.d.d.a.length-1;D>=0;D--)L=lt._3(this.d,D),L.i!=(tt.AI(),tt.zI)?lt._3(this.d,D).i==tt.xI&&lt.b4(this.d,D):L.i=tt.yI;for(this.b=z(l.d7,H,6,t.f,15,1),i=0;i<t.f;i++)for(Q=0;Q<e.f;Q++)if(j.Math.abs(e.w[Q])==j.Math.abs(t.w[i])){r=(h=n.om(t,i),h+n.mm(t,i,h,t.C[i])-n.up(t,i)-(s=n.om(e,Q),s+n.mm(e,Q,s,e.C[Q])-n.up(e,Q))),this.b[i]=r>0?r:0;break;}this.e=new n.hu();},x(285,1,{},tt.nI),l.z9=X(285),tt.oI=function(t,e,A){var P,r,i,D;if(P=A[t.a],r=A[t.b],t.i==(tt.AI(),tt.wI))return n.Vk(e,P,r,t.g),true;if(i=n.Wo(e,P,r),t.i==tt.uI)return n.Ln(e,i,t.g),true;if(t.i==tt.vI){if(D=n.em(e,i)+t.f,D>=0&&D<=3)return n.Ln(e,i,(n.Qk(),D==0?16:D==1?1:D==2?386:D==3?4:D==4?32:64)),true;}else if(t.i==tt.yI)return e.H[i]=512,true;return  false;},tt.pI=function(t,e){return t.d!=e.d?t.d<e.d?-1:1:t.e!=e.e?t.e<e.e?-1:1:0;},tt.qI=function(t,e){var A,P;t.c==e.c?t.i=(tt.AI(),tt.xI):Pn(e.c)==1?(t.g=e.c==1?1:e.c==2?386:e.c==4?4:e.c==16?16:8,t.i=(tt.AI(),tt.uI)):(A=(t.c&16)!=0?0:(t.c&1)!=0?1:(t.c&2)!=0?2:3,P=(e.c&16)!=0?0:(e.c&1)!=0?1:(e.c&2)!=0?2:3,P==A?t.i=(tt.AI(),tt.xI):(t.f=P-A,t.i=(tt.AI(),tt.vI)));},tt.rI=function(t,e){var A,P,r;this.a=t.D[0][e],this.b=t.D[1][e],P=n.Kl(t,this.a),r=n.Kl(t,this.b),P<r?(this.d=P,this.e=r):(this.d=r,this.e=P),this.c=t.G[e]&31,A=t.H[e]&127,A==16?this.c|=16:A==2?this.c|=2:A==4?this.c|=4:this.c|=1,this.i=(tt.AI(),tt.zI);},x(182,1,{182:1,32:1},tt.rI),y.Rb=function(t){return tt.pI(this,t);},y.lb=function(t){return this.d==t.d&&this.e==t.e;},y.a=0,y.b=0,y.c=0,y.d=0,y.e=0,y.f=0,y.g=0,l.y9=X(182),tt.AI=function(){tt.AI=o1,tt.wI=new tt.BI("CREATE",0),tt.vI=new tt.BI("CHANGE_DIF",1),tt.uI=new tt.BI("CHANGE_ABS",2),tt.yI=new tt.BI("REMOVE",3),tt.xI=new tt.BI("NO_CHANGE",4),tt.zI=new tt.BI("UNKNOWN",5);},tt.BI=function(t,e){FA.call(this,t,e);},tt.CI=function(){return tt.AI(),F(k(l.x9,1),s0,98,0,[tt.wI,tt.vI,tt.uI,tt.yI,tt.xI,tt.zI]);},x(98,56,{98:1,4:1,32:1,56:1},tt.BI),l.x9=Xe(98,tt.CI),a.DI=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(n.aw(e,1),c=t.k,g=0,d=c.length;g<d;++g)b=c[g],Q=A[b.a],L=A[b.b],s=n.Wo(e,Q,L),s==-1?n.Vk(e,Q,L,b.e):b.e==-1?e.H[s]=512:b.e!=-2&&n.Ln(e,s,b.e);if(u=n.ul(e),t.a.length!=0)for(n.aw(e,7),r=t.a,i=0,D=r.length;i<D;++i)P=r[i],o=u==null?A[P]:u[A[P]],h=e.u[o]&3,n.tn(e,o,h==1?2:1,false),n.vq(e,o);return u;},a.EI=function(t){var e,A,P,r,i;for(i=new a.aJ(t.g,t.f),r=z(l.d7,H,6,t.g.q,15,1),A=0;A<t.g.q;A++)r[A]=n.Kl(t.g,A);for(P=z(l.d7,H,6,t.f.q,15,1),e=0;e<t.f.q;e++)P[e]=n.Kl(t.f,e);return -a._I(i,a.UI(i,r,P))*.25;},a.FI=function(t,e){var A,P,r,i,D;for(t.i=z(l.d7,H,6,t.g.f,15,1),P=0;P<t.f.f;P++)n.Kl(t.f,P)!=0&&(t.i[e[n.Kl(t.f,P)]]=n.lw(t.f,P));if(D=z(l.d7,H,6,t.g.q,15,1),i=n.gp(t.g,D,false,false),i>1)for(r=z(l.d7,H,6,i,15,1),A=0;A<t.g.f;A++)t.i[A]|=r[D[A]]++<<5;},a.GI=function(t,e,A,P,r){var i,D,u,o;if(t.j.length==0||t.n.a.length==0)return null;for(o=z(l.zeb,l1,12,e,0,2),u=z(l.zeb,ht,6,P,16,1),D=0;D<r.length;D++)r[D]!=-1&&n.Kl(t.f,D)==0&&(u[r[D]]=true);for(i=0;i<A.length;i++)A[i]!=-1&&n.Kl(t.g,i)==0&&(o[A[i]]=u);return o;},a.HI=function(t,e,A){var P,r,i,D,u,o,Q;if(o=false,n.Po(t.g,e)==0)for(u=1;u<n.Zo(t.g,e);u++)for(Q=0;Q<u;Q++)P=n.Yo(t.g,e,u),r=n.Yo(t.g,e,Q),i=n.Kl(t.g,P),D=n.Kl(t.g,r),A[i]>A[D]^P>r&&(o=!o);return o;},a.II=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N;for(this.d=t,N=tt.TH(e,false,null),(N.g.a.length!=1||N.f.a.length!=1)&&T.Ujb(),N.c||tt.fH(N)||T.Ujb(),this.g=U.wj(N.g,0),this.f=U.wj(N.f,0),n.aw(this.g,1),n.aw(this.f,63),Q=new lt.d4(),this.c=z(l.d7,H,6,this.g.f+1,15,1),this.c[0]=-1,P=0;P<this.g.f;P++)n.Kl(this.g,P)!=0&&(this.c[n.Kl(this.g,P)]=P);for(a.FI(this,this.c),V=z(l.zeb,ht,6,this.g.g,16,1),m=0;m<this.f.g;m++)g=n.Yl(this.f,0,m),d=n.Yl(this.f,1,m),h=n.Kl(this.f,g),b=n.Kl(this.f,d),h!=0&&b!=0&&(r=this.c[h],i=this.c[b],C=n.im(this.f,m),S=n.Wo(this.g,r,i),S==-1?lt.Z3(Q,new a.KI(r,i,h,b,C)):((n.gm(this.g,S)&31)==0&&(O=n.im(this.g,S),O!=C&&lt.Z3(Q,new a.KI(r,i,h,b,C))),V[S]=true));for(o=0;o<this.g.g;o++)V[o]||(r=n.Yl(this.g,0,o),i=n.Yl(this.g,1,o),!n.Im(this.g,r)&&!n.Im(this.g,i)&&(h=n.Kl(this.g,r),b=n.Kl(this.g,i),(h!=0||b!=0)&&lt.Z3(Q,new a.KI(r,i,h,b,-1))));for(this.k=lt.c4(Q,z(l.A9,{463:1,4:1,5:1,11:1,8:1},147,0,0,1)),L=0,D=0;D<this.g.f;D++)n.Kl(this.g,D)==0&&!n.Im(this.g,D)&&++L;for(this.j=z(l.d7,H,6,L,15,1),L=0,u=0;u<this.g.f;u++)n.Kl(this.g,u)==0&&!n.Im(this.g,u)&&(this.j[L++]=u);for(this.n=new U.Lj(),s=z(l.zeb,ht,6,this.f.f,16,1),A=0;A<this.f.f;A++)n.Kl(this.f,A)==0&&!n.Im(this.f,A)&&!s[A]&&U.rj(this.n,new a.UJ(this.f,A,s));for(this.b=this.j.length==0&&this.n.a.length==0,this.a=z(l.d7,H,6,0,15,1),c=0;c<this.f.f;c++)v=n.Ml(this.f,c),(v==1||v==2)&&(M=this.c[n.Kl(this.f,c)],M!=-1&&(K=n.Ml(this.g,M),a.HI(this,c,this.c)==(K==v)&&(this.a=U.Eqb(this.a,this.a.length+1),this.a[this.a.length-1]=M)));this.e=a.EI(this);},x(199,1,{199:1},a.II),y.b=false,y.e=0,l.C9=X(199),a.JI=function(t,e){return t.c<e.c?-1:t.c>e.c?1:t.d<e.d?-1:t.d>e.d?1:0;},a.KI=function(t,e,A,P,r){A<P?(this.a=t,this.b=e,this.c=A,this.d=P):(this.a=e,this.b=t,this.c=P,this.d=A),this.e=r;},x(147,1,{147:1,32:1},a.KI),y.Rb=function(t){return a.JI(this,t);},y.a=0,y.b=0,y.c=0,y.d=0,y.e=0,l.A9=X(147),a.OI=function(){a.OI=o1,a.MI=F(k(l.Sbb,2),c0,13,0,[F(k(l.Sbb,1),st,2,6,["e","daXJ@@PjyITuiX@`!dahJ@@SJYIMMfPB#IaLJfxP IaAhl[`#!BDpAl@AL@[@Be}aL@[@@ !B|Osp?QZR?O_}}zGze`@##"]),F(k(l.Sbb,1),st,2,6,["g","gBa`@lde@XS@!gCA`@mDPOdku`#qbq qJt#!B[G|S_qgq !BjW}q]cga##"]),F(k(l.Sbb,1),st,2,6,["o","gGQHDDqIUjdB!gGQHHDqAeMjhB#qbqk qfQ]#!B@AL@[@@S@Fp@Dp !BFQ{~_q|ZGvUSYp##"]),F(k(l.Sbb,1),st,2,6,["r","gOQdEjHbAFQRBrtAaJ!gNQdEbHbABCTKPFDH#qbMwX qbM~X#!BNDm`ra?UjW~YBYX@ !Ba[zw?_x@?g~H?XO~##"]),F(k(l.Sbb,1),st,2,6,["s","gOp`ATigujj`H!gOp`ATiVKjj`H#qnyS` qY~eP#!BTLtk^sE?BOs|]pc} !BbOvw?_y??g~H?[_}##"]),F(k(l.Sbb,1),st,2,6,["t","gOP`@dcUZdB!gNp`@tiTMjj@`#q{ir` qkLrx#!Be`Bzr_wp?OC}|Osp !B?g~w@k_}m?vw@n[a##"]),F(k(l.Sbb,1),st,2,6,["u","daXB`Hrn@HrPEILt`D!daXB`DJn@HRUMjV@H#IxVaLJ` IylJhPP#!B[G}l@OKyDpAl@AL@[@@ !B@Fp@DpAl@AN]?`Al@AL##"]),F(k(l.Sbb,1),st,2,6,["Sakurai","gOQH@wAINvZ@pdcFe@x@!gOQH@wAIgJi@pdcFe@x@#qreKx qrLkx#!BDpAl@IknDw|S@Fp@ !Bb@JH?_x@b@JH?Ven##"]),F(k(l.Sbb,1),st,2,6,["Mitsunobu","gFP`ATfRhdPp`}KEYg]d@!gFP`ATfRhd`pekL{l`#qrLk qZLn#!B@hc}b@C~@h`YM` !B@hc}b@C~@h`YM`##"]),F(k(l.Sbb,1),st,2,6,["Aldol-Addition","gOQ@AdTAcS@[Q^crJTLES`DJsL?vH!gGQ@@dsuRAcJg@HUaX#qYEbp qYub#!BxOyBzLKg`dG~xG~{ !Bb@K~@Hc}FBIA@@##"]),F(k(l.Sbb,1),st,2,6,["Aldol-Condensation","gOQ@AdTAcS@[Q^crJTLES`DJsL?vH!gFQ@@`rrpdlHHpipBEXf@#qYEbp q^aU#!B{ZRRqA?AQfyA@L_C !B}QFw@h`B_tnH_P##"]),F(k(l.Sbb,1),st,2,6,["Acetal-Aldol-Addition","dmdB@@serQS@sJjfd@p`Xir\\@`j\\@aUJXK@!daxL@@[df[ZjT@qQdxACdxABjTqf@#qB@`OuX qBtM{#!B_]]}rHKWw?y}uy[~GJbBu{wWqG| !BfJK~TIa]fJJghg{`pP@##"]),F(k(l.Sbb,1),st,2,6,["Acetal-Aldol-Condensation","dmdB@@serQS@sJjfd@p`Xir\\@`j\\@aUJXK@!gNp`CTjUiV@qQS`DJg@HUVXV@#qB@`OuX qqj{`#!B?[_}b@Jw_?{}m~[~[N@Bm?vwkN| !BfsK~yzPrw}m?rzQM##"]),F(k(l.Sbb,1),st,2,6,["Acetal-Aldol-Condensation-Cyclization","dkLB@@ZURYUvUjljh@paHpr\\@`!didD@@yIfUXXHL@CFNS`D@#IXljNPY@@@ IXljIyA#!BbOw~_x`Bm?vH?[_}b@JH?_y?b@Jw?Xc} !BbOvH?Oy??`BH?Xa}?`C~_p##"]),F(k(l.Sbb,1),st,2,6,["Enolester-Cleavage","gOQ`@fdscT`_Qp!gOQ`@cdTASS@P#q}Kr` q}cNP#!B@k]}mpC~@k]}mqdY !Bb@K~@Hc}BfzH@hc}##"]),F(k(l.Sbb,1),st,2,6,["propargylEnone","gCa@@dmXFD@!gCa@@dkHD#qNT qLV#!BbOw~_?y? !B@AL@[@@S##"]),F(k(l.Sbb,1),st,2,6,["Arndt-Eistert","daiDaJYBBHj^{HhAYMpAaA@!daiD`FzLBHPVsZl@p`@#IyHHZ[@ IzDGBi`#!B?`Bw?H`Bn]{~\\g?~@Ox !B@rzH?_y?b@JH?_n~bOt##"]),F(k(l.Sbb,1),st,2,6,["Curtius","gO]IcVaDF[s{HhCIe@`!gN]HMWADHJfm`XP@#q~Jk` qytUX#!B@O{|b@Jw\\o{~_?x@ !Bj]y??g?~?[^G_hc}##"]),F(k(l.Sbb,1),st,2,6,["diazomethanHomologation","gFU@lQioIIs\\AyH!gFU@CPdimXD#qbM^ qbqk#!B?X`BbFZN?`C~_p !B@AL@[@@Su[x@Dp##"]),F(k(l.Sbb,1),st,2,6,[wD,"deD@@LdbEdSTu@FqHWSDda`JFChXIa`?tdKi@!deD@@Ldb`\\SUM@FqHPsDda`JF@XXIa`?tdHY@#qTEApX qQECf@#!BQzK~}ubbW`BEgcb]?a@gg[zO !BQzK~}ubbW`Ag{VVAQzJ~c?xP##"]),F(k(l.Sbb,1),st,2,6,[wD,"deD@@LdbEdSTu@FqHWSDda`JFChXIa`?tdKi@!gC`@DiZDE@#qPDA@p qQf#!BmpK~_x`Bm?tAs[]}?`BH_[_| !B_vp@[G|S##"]),F(k(l.Sbb,1),st,2,6,["yne-Metathesis","daX@@LdPLWWPAlRXwQIHXLa`ZFChXO}IL[`!daX@@LdPLWWPAlRXwQIHXLa`ZFChXO}IL[`#qT@q@ qt@Q@#!BZmoi@Fjo|SFe|IkGiU@ !BZmoi@Fjo|SFe|IkGiU@##"]),F(k(l.Sbb,1),st,2,6,["ene-yne-Metathesis","dcd@@LdPLPLWSSPAlrXwKlVRFCHXFa`zFAXXMa`?udqnWP!dcT@@LdbbplTsTt@[MeMr{Ela`jFAhXNa`VFCXXO}[J[et#qe@N@S@ qeHP@s@#!B?g?~@Oy?^gG}bOvw?H`E@PJw@hc}mp !B?`BH?[_}mpJH@oy??`AuC`Jw@hc}mp##"]),F(k(l.Sbb,1),st,2,6,["Alkyne-Cyclisation","gG`@DcO|CFAFC`@!gFp@DiTt@@CFAFC`@#qi\\V qiSt#!B_y[qA`Biu^zV@@ !B?g~w@k_}m?vw@`##"]),F(k(l.Sbb,1),st,2,6,["Elimination-Claisen","gNp`AldTQji@~a`!gGQ@@djmRA@#qtHUX qtSi#!Bm?vH?[\\B?g~H@hc} !B@AL@[@@S@Fp@Dp##"]),F(k(l.Sbb,1),st,2,6,["imineFormationAzaCope","daZH@LAIMUjd@pRL@!daZH@HAAn]jd@p`@#IGfaLJ` IFDzfK@#!BDpAl@IkqDpAl@AL@[@@ !BFaFw@h`BbOw~@H`BbOt##"]),F(k(l.Sbb,1),st,2,6,["didehydroCopeWithAromatisation","gNp@Di]ejDcjcr|wK`!gOp@DjWkB@@H#qrLkx q\\oQp#!B?g~H?K_}bGvH?H`B !Bm?w~@Hc}mpJw@ox@##"]),F(k(l.Sbb,1),st,2,6,["symAldolNitrogenRing","dovJ@GBfttf\\v\\qjViPCADGbDodnGp!doNJ@JCSmtefWTCaYjje@H#IlZXi]]yL~C IqMVCzaIim?#!BQtl_riY?Qtl_rfuvNCQ`uZd@NCQ`uVVu}?sA]P !B?`BH@ox@bGvH@k\\Bb@JH_Xa}b@K~_rYltUr|W@##"]),F(k(l.Sbb,1),st,2,6,["Diels-Alder","gFP@LdPLjA@!gFp@DiTujhB#qiZf qMSf#!B?_C}}?spIPFV@@ !B?g~w@k_}m?vw@`##"]),F(k(l.Sbb,1),st,2,6,["Claisen-RA","gGQ@@eKtRA@!gGQ@@djqRA@#qMsT qM\\V#!B_qL@Dw}l@Fp@Dp !B@AL@[@@S@Fp@Dp##"]),F(k(l.Sbb,1),st,2,6,["Cope","gGQ@DeZmRAbhcApIF@P@!gGQ@HeZmRAbhc@HIFC`@#qkNT qi\\V#!B_vp@[@@S@Fp@Dp !B_vp@[@@S@Fp@Dp##"]),F(k(l.Sbb,1),st,2,6,["OxyCope","gNq@@dr}SHFD@!gNq@@djkUHD#qynZ` qykbp#!B?g~w?Xc}mpJH@hc} !B@Fp@DpAl@AL@[@@S##"]),F(k(l.Sbb,1),st,2,6,["Vinylcyclopropane","gKP@DklVj@`!gKP@Di\\Vj@`#qaVh qTqh#!Bm?vH?PC~?K\\ !B?g|_Fa}eTv\\##"]),F(k(l.Sbb,1),st,2,6,["Furan-CH2-OH-Rearrangement","gOp`@tiguif`H!gOp`@tfWMZZ`H#qZna@ qZtM@#!BTLtk^sE?BOs|]pc} !BBOuPtdy?UGm@V]Ho##"]),F(k(l.Sbb,1),st,2,6,["rearrangement1032","gOp`ATieMji`H!gOp`ATeekZj`H#qaSnx qa]~P#!BTLtk^pc|LW?|]pc} !BBOpH?UCRSg}T^tAY##"]),F(k(l.Sbb,1),st,2,6,["Pinacol-Rearrangement","gNp`ATffjj@pPh!gOP`@tfXZhCAA`#qb^ix qb^oH#!B@k^H@k_}@k_~@Hc} !BbOvH@oy??`BH?PFf##"]),F(k(l.Sbb,1),st,2,6,["1,3-WaterEliminationWith12Shift","gJQ@@dmU@_SP!gKa@@`vtA}A@#qiTp qjap#!BbOvH@ox@bOt !BJdE?[@Al@AL##"]),F(k(l.Sbb,1),st,2,6,["Epoxydation","gB``ADcdCB@!gC``AhtUPGtt@#qqb qtQ#!BjW}Y\\YX@ !B?g~w?^Va##"]),F(k(l.Sbb,1),st,2,6,["oxydativePropargylAmine13Shift","gKi@HDEZpLHOQP!gJY@BDeVXB#qMr` qNTh#!BqiXTy{U?mW| !B@Fp@DpAl@AL##"]),F(k(l.Sbb,1),st,2,6,["Baeyer-Villiger","gFQ`@[dTAZ`LHP!gFQ`@jdrMPGtl@#qrak qrlK#!B_?{}mwvHs^FVP@ !BbOvH@oy?bOuQzP##"]),F(k(l.Sbb,1),st,2,6,[gD,"gGa@@dYs@XHFJIBDQbUeHPbLRl@!gFQ@@eNUPFJIBDQbUeHPbLRls@`#qbq] qfQk#!B@AOIDW}l@tD@Dp !B_qL@Dw}l_qNcDP##"]),F(k(l.Sbb,1),st,2,6,[gD,"gBa@@d\\`XP@!gJQ@@eOU@XpdHQFIVY`P#qbq qfQ@#!B@AOIDW}l !B_qL@Dw}l_qL##"]),F(k(l.Sbb,1),st,2,6,["Hantzsch Thiazol","daZHPDp@chaIMefh@ppDzTD~hYmC^bhbcPp]dQbUg~pp!gKXHL@aJWFe`H#qNPe@ qNj`#!BvuK[KUxv_yS[k_zhvuH !BTqa`FbpX?`@##"]),F(k(l.Sbb,1),st,2,6,["Oxadiazole","gOX`BEdTASW@XQ@!gOu@HPeKNMKTA@#qrDMX qpULX#!BmpK~@K_}Mlx@?`C~ !BZ?`C}v|m_rYR[z?\\##"]),F(k(l.Sbb,1),st,2,6,["Imidazole","dmeHPNg@qJqLbTtATijZ@LLJnuDmhWtSDXUFC`?rIoTAP!gOt@ATieuej`H#qDPpM_@ q~ZM`#!BqvKGg_yOqvKGg_xphrGkLcz@_sD !BTMHkACD@BOw|B@QT##"]),F(k(l.Sbb,1),st,2,6,["1,2,3-Triazole","gB`HAbIGXFDWiM@!gF|@ADeKXmT`P#QIp Q@v#!B_vpU?g}l !BTv]`YRqg?g|XK@##"]),F(k(l.Sbb,1),st,2,6,["1,2,4-Triazole","deFD@NALbbfASUW@FD]YJZLUCAVJ}?nES@!gO|@ABeKNLuRA@#qDB@FM q@LuP#!BY?r~@F_]jDJW`j`}Iaxx[UC] !BTv]@IPqgog|hCBT_##"])]);},a.QI=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt;for(a.TI(),o=new a.jK(),a.gK(o,e),m=o.j,h=o.d,n.aw(m,1),n.aw(h,1),S=new n.hu(),d=new n.hu(),n.bu(S,m),n.bu(d,h),t.c=P1,D=null,i=null,r=0,I=0,t.a=new T.Pjb(),A=new n.sw(),O=a.NI,K=0,N=O.length;K<N&&(V=O[K],I++!=t.b);++K)if(n.$t(S,V.g),n._t(S,V.i),n.Ot(S,6,8)!=0&&(n.$t(d,V.f),n.Ot(d,2,8)!=0&&S.F.a.length<=512)){for(u=-1e4,g=U.wj(d.F,0),M=new U.wqb(S.F);M.a<M.c.a.length&&(v=U.vqb(M),!(I++>=t.b));)n.nl(m,A),Q=a.DI(V,A,v),Dt=a.GI(V,m.f,v,h.f,g),n.aw(A,1),P=z(l.d7,H,6,A.f,15,1),c=z(l.d7,H,6,h.f,15,1),a.dK(o,A,h,P,c,Dt),Qt=o.q-V.e,u<Qt&&(u=Qt),t.c<Qt&&(t.c=Qt,D=a.RI(P,c,Q,m.f),i=c,r=o.b);s=o.a<=1?"":zD+o.a+")",T.Njb(t.a,V.d+(V.b?"_S":V.j.length==0?"_L":V.n.a.length==0?"_I":"_U")+(lt.Q3(),lt.R3(u,5))+s+p1);}C=z(l.d7,H,6,m.f,15,1),b=z(l.d7,H,6,h.f,15,1),a.dK(o,m,h,C,b,null),ut=o.q,t.c<=ut&&(t.c=ut,D=C,i=b,r=o.b),L=o.a<=1?"":zD+o.a+")",T.Njb(t.a,(lt.Q3(),"no rule:"+lt.R3(ut,5)+L+p1)),t.c!=P1&&a.XJ(o,e,D,i,r);},a.RI=function(t,e,A,P){var r,i,D,u,o;for(D=z(l.zeb,ht,6,t.length+1,16,1),o=z(l.d7,H,6,P,15,1),u=0;u<P;u++)r=A==null?u:A[u],r!=-1&&(o[u]=t[r],D[o[u]]=true);for(i=0;i<e.length;i++)D[e[i]]||(e[i]=0);return o;},a.SI=function(){a.OI(),this.b=St;},a.TI=function(){var t,e;if(a.NI==null&&a.NI==null){for(t=z(l.C9,C1,199,a.MI.length,0,1),e=0;e<a.MI.length;e++)t[e]=new a.II(a.MI[e][0],a.MI[e][1]);a.NI=t;}},x(294,1,{},a.SI),y.b=0,y.c=0,l.B9=X(294),a.UI=function(t,e,A){var P,r,i,D;for(i=z(l.d7,H,6,t.b.f+1,15,1),U.Oqb(i,i.length,-1),r=0;r<A.length;r++)A[r]!=0&&(i[A[r]]=r);for(D=z(l.d7,H,6,t.b.f,15,1),U.Oqb(D,D.length,-1),P=0;P<e.length;P++)D[P]=i[e[P]];return D;},a.VI=function(t,e){var A,P,r,i;return A=t.D[0][e],P=t.D[1][e],r=n.Gm(t,A),i=n.Gm(t,P),!r&&!i?e<t.g&&n.ot(t.p,e)?x9:Br+n.em(t,e)/10:r&&i||r&&n.Km(t,P)||i&&n.Km(t,A)?bA:r&&a.nK(t,P,A)||i&&a.nK(t,A,P)?oP:r&&a.lK(t,P,A)||i&&a.lK(t,A,P)?Hr:r&&P<t.f&&n.nt(t.p,P)||i&&A<t.f&&n.nt(t.p,A)?1.9500000476837158:Br;},a.WI=function(t,e){return e<t.g&&(n.tt(t.p,e)||t.H[e]==8)?0:t.H[e]&127;},a.XI=function(t,e){return e<t.g&&(n.tt(t.p,e)||t.H[e]==8)?1.5:n.em(t,e);},a.YI=function(t,e,A){var P,r,i;if(P=A[e],P!=-1&&n.Ml(t.a,P)!=0&&a.ZI(t,e,P,A)){if(i=n.Ml(t.b,e),r=n.Ml(t.a,P),i==3){if(r==1||r==2)return 5;}else if(r==3||a.$I(t,e,A)==(i==r))return 5;}return 0;},a.ZI=function(t,e,A,P){var r,i,D,u;if(n.Zo(t.b,e)!=n.Zo(t.a,A))return  false;for(i=0;i<n.Zo(t.b,e);i++){for(u=n.Yo(t.b,e,i),r=false,D=0;D<n.Zo(t.a,A);D++)if(P[u]==n.Yo(t.a,A,D)){r=true;break;}if(!r)return  false;}return  true;},a.$I=function(t,e,A){var P,r,i,D,u,o,Q;if(o=false,n.Po(t.b,e)==0)for(u=1;u<n.Zo(t.b,e);u++)for(Q=0;Q<u;Q++)P=n.Yo(t.b,e,u),r=n.Yo(t.b,e,Q),i=A[P],D=A[r],i!=-1&&D!=-1&&i>D^P>r&&(o=!o);return o;},a._I=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(s=0,D=z(l.zeb,ht,6,t.a.f,16,1),P=e,r=0,i=P.length;r<i;++r)A=P[r],A!=-1&&(D[A]=true);for(h=z(l.zeb,ht,6,t.a.g,16,1),d=0;d<t.b.g;d++)if(c=n.Yl(t.b,0,d),g=n.Yl(t.b,1,d),!(t.b.K&&(n.Im(t.b,c)||n.Im(t.b,g)))){if(u=e[c],o=e[g],a.XI(t.b,d),u==-1||o==-1){(u!=-1||o!=-1)&&(s+=a.VI(t.b,d));continue;}if(L=n.Wo(t.a,u,o),L==-1){s+=a.VI(t.b,d);continue;}h[L]=true,s+=a.WI(t.b,d)==a.WI(t.a,L)?0:1;}for(Q=0;Q<t.a.g;Q++)t.a.K&&(n.Im(t.a,n.Yl(t.a,0,Q))||n.Im(t.a,n.Yl(t.a,1,Q)))||h[Q]||(s+=a.VI(t.a,Q));for(b=0;b<t.b.f;b++)n.Ml(t.b,b)!=0&&(s+=a.YI(t,b,e));return -s;},a.aJ=function(t,e){this.b=t,this.a=e;},x(202,1,{},a.aJ),l.D9=X(202),a.bJ=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;if(m=new a.aJ(t.e,t.c),g=a.UI(m,t.f,t.d),t.a.a.length==0)return a._I(m,g);for(C=1,M=new U.wqb(t.a);M.a<M.c.a.length;)v=U.vqb(M),C*=a.hJ(v);if(C>4e6)return T.Ujb(),-C;for(e=0,D=z(l.d7,H,6,t.a.a.length,15,1),b=z(l.d7,H,6,t.a.a.length,15,1),o=0;o<t.a.a.length;o++)v=U.wj(t.a,o),b[o]=a.kJ(v),D[o]=e,e+=v.a;for(i=Er,r=null,P=null,c=z(l.d7,H,6,t.a.a.length,15,1),h=t.a.a.length!=0;h;){if(S=false,t.i!=null){for(Q=0;Q<t.a.a.length;Q++)if(a.lJ(U.wj(t.a,Q),c[Q],t.i)){S=true;break;}}if(!S){for(Q=0;Q<t.a.a.length;Q++)a.gJ(U.wj(t.a,Q),c[Q],g);if(d=a._I(m,g),i<d)for(i=d,r=z(l.d7,H,6,e,15,1),P=z(l.d7,H,6,e,15,1),A=0,L=0;L<t.a.a.length;L++)v=U.wj(t.a,L),a.jJ(v,c[L],r,A),a.iJ(v,c[L],P,A),A+=v.a;}for(h=false,u=0;u<c.length;u++){if(++c[u],c[u]<b[u]){h=true;break;}c[u]=0;}}if(i!=Er)for(s=t.b,u=0;u<e;u++)++s,t.f[r[u]]=s,t.d[P[u]]=s;return i;},a.cJ=function(t,e,A,P,r,i){var D,u,o,Q,L,s,h,b;for(this.e=t,this.c=e,this.f=A,this.d=P,this.g=r,this.b=r,this.i=i,Q=new U.awb(),u=0;u<t.f;u++)A[u]==0&&(o=t.C[u]+(t.A[u]<<16),h=U.jpb(Q,T.uib(o)),h||(h=new a.oJ(this),U.Uvb(Q,T.uib(o),h)),h.d=a.eJ(u,h.d),h.d.length<=h.c.length&&(h.a=h.d.length));for(D=0;D<e.f;D++)P[D]==0&&(o=e.C[D]+(e.A[D]<<16),h=U.jpb(Q,T.uib(o)),h||(h=new a.oJ(this),U.Uvb(Q,T.uib(o),h)),h.c=a.eJ(D,h.c),h.c.length<=h.d.length&&(h.a=h.c.length));for(this.a=new U.Lj(),b=(s=new U.Job(Q).a.ad().Sb(),new U.Qob(s));b.a.cd();)h=(L=b.a.dd(),L.gd()),a.mJ(h)||U.rj(this.a,h);},x(399,1,{},a.cJ),y.b=0,y.g=0,l.F9=X(399),a.dJ=function(t){t.d=z(l.d7,H,6,0,15,1),t.c=z(l.d7,H,6,0,15,1);},a.eJ=function(t,e){var A,P;for(P=z(l.d7,H,6,e.length+1,15,1),A=0;A<e.length;A++)P[A]=e[A];return P[e.length]=t,P;},a.fJ=function(t,e){var A,P,r,i,D,u;for(P=t,r=0,i=P.length;r<i;++r)if(A=P[r],e.j[A]!=0)return  false;for(D=n.Al(e,t[0]),u=1;u<t.length;u++)if(n.Al(e,t[u])!=D)return  false;return  true;},a.gJ=function(t,e,A){var P,r,i,D,u,o;if(o=U.wj(t.b,e),t.d.length<=t.c.length)for(u=0;u<t.a;u++)A[t.d[u]]=t.c[o[u]];else {for(r=t.d,i=0,D=r.length;i<D;++i)P=r[i],A[P]=-1;for(u=0;u<t.a;u++)A[t.d[o[u]]]=t.c[u];}},a.hJ=function(t){var e,A,P;for(P=j.Math.max(t.d.length,t.c.length),A=1,e=P-t.a+1;e<=P;e++)A*=e;return A;},a.iJ=function(t,e,A,P){var r,i;if(t.d.length>t.c.length)for(r=0;r<t.a;r++)A[P+r]=t.c[r];else for(i=U.wj(t.b,e),r=0;r<t.a;r++)A[P+r]=t.c[i[r]];},a.jJ=function(t,e,A,P){var r,i;if(t.d.length<=t.c.length)for(r=0;r<t.a;r++)A[P+r]=t.d[r];else for(i=U.wj(t.b,e),r=0;r<t.a;r++)A[P+r]=t.d[i[r]];},a.kJ=function(t){var e,A;return t.b=new U.Lj(),A=z(l.d7,H,6,t.a,15,1),e=z(l.zeb,ht,6,j.Math.max(t.d.length,t.c.length),16,1),a.nJ(t,0,e,A),t.b.a.length;},a.lJ=function(t,e,A){var P,r;if(r=U.wj(t.b,e),t.d.length<=t.c.length){for(P=0;P<t.a;P++)if(A[t.d[P]]!=null&&A[t.d[P]][t.c[r[P]]])return  true;}else for(P=0;P<t.a;P++)if(A[t.d[r[P]]]!=null&&A[t.d[r[P]]][t.c[P]])return  true;return  false;},a.mJ=function(t){var e,A,P;if(t.a==0)return  true;if(t.d.length==1&&t.c.length==1)return (t.e.i==null||t.e.i[t.d[0]]==null||!t.e.i[t.d[0]][t.c[0]])&&(++t.e.b,t.e.f[t.d[0]]=t.e.b,t.e.d[t.c[0]]=t.e.b),true;if(a.fJ(t.d,t.e.e)||a.fJ(t.c,t.e.c)){for(e=j.Math.min(t.d.length,t.c.length),P=0;e!=0&&P<t.d.length;P++)for(A=0;A<t.c.length;A++)if(t.e.d[t.c[A]]==0&&(t.e.i==null||t.e.i[t.d[P]]==null||!t.e.i[t.d[P]][t.c[A]])){--e,++t.e.b,t.e.f[t.d[P]]=t.e.b,t.e.d[t.c[A]]=t.e.b;break;}return  true;}return  false;},a.nJ=function(t,e,A,P){var r;for(r=0;r<A.length;r++)A[r]||(A[r]=true,P[e]=r,e+1==P.length?U.rj(t.b,U.Eqb(P,P.length)):a.nJ(t,e+1,A,P),A[r]=false);},a.oJ=function(t){this.e=t,a.dJ(this);},x(201,1,{201:1},a.oJ),y.a=0,l.E9=X(201),a.pJ=function(t,e){return t.b<e.b?-1:t.b>e.b?1:t.a<e.a?-1:t.a>e.a?1:0;},a.qJ=function(t,e){this.b=t,this.a=e;},x(126,1,{126:1,32:1},a.qJ),y.Rb=function(t){return a.pJ(this,t);},y.a=0,y.b=0,l.J9=X(126),a.sJ=function(t,e){var A;return A=a.wJ(t.a,e),t.a=a.vJ(t.a,A),A;},a.tJ=function(t){t.a=t.b;},a.uJ=function(){this.b=new a.yJ(),this.a=this.b;},x(398,1,{},a.uJ),l.G9=X(398),a.vJ=function(t,e){return t.a[e];},a.wJ=function(t,e){if(t.b==-1)t.a=z(l.H9,C1,169,e,0,1),t.b=e,t.c=e-1;else do t.c=t.c==0?t.b-1:t.c-1;while(t.a[t.c]&&a.xJ(t.a[t.c]));return !t.a[t.c]&&(t.a[t.c]=new a.yJ()),t.c;},a.xJ=function(t){var e,A,P,r,i;if(!t.d)if(t.b==-1)t.d=true;else {for(i=false,A=t.a,P=0,r=A.length;P<r;++P)if(e=A[P],!e)i=true;else if(!a.xJ(e))return  false;t.d=!i;}return t.d;},a.yJ=function(){this.b=-1;},x(169,1,{169:1},a.yJ),y.b=0,y.c=0,y.d=false,l.H9=X(169),a.zJ=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h,b;for(b=z(l.zeb,ht,6,P.length,16,1),u=t,Q=0,s=u.length;Q<s;++Q)if(i=u[Q],e[i]==0){if(P[A[i]]||b[A[i]])return  false;b[A[i]]=true;}for(h=-1,D=t,o=0,L=D.length;o<L;++o)if(i=D[o],e[i]==0){if(h==-1)h=r[i];else if(r[i]!=h)return  false;}return  true;},a.AJ=function(t){var e,A,P,r,i,D,u,o;for(t.t=new U.Lj(),P=1,i=0,A=0;A<t.u.f;A++)n.Kl(t.u,A)!=0&&!n.zm(t.u,A)&&(i=j.Math.max(i,n.Kl(t.u,A)));if(i!=0){for(r=z(l.d7,H,6,i+1,15,1),e=0;e<t.u.f;e++)n.Kl(t.u,e)!=0&&!n.zm(t.u,e)&&(r[n.Kl(t.u,e)]=e+1);for(u=0;u<t.D.f;u++)o=n.Kl(t.D,u),o!=0&&o<=i&&!n.zm(t.D,u)&&r[o]!=0&&(D=r[o]-1,t.I[u]=P,t.B[D]=P++,U.rj(t.t,new a.qJ(u,D)));}return t.t.a.length;},a.BJ=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b;for(s=a.EJ(t.D),u=a.EJ(t.u),D=z(l.mdb,l1,44,8,0,1),b=a.CJ(t.D,s),Q=a.CJ(t.u,u),L=0;L<=7;L++)for(D[L]=new U.bwb(new lt.E3()),i=(P=new U.kwb(new U.qwb(new U.xpb(b[L]).a).b),new U.Gpb(P));U.fob(i.a.a);)r=(A=U.iwb(i.a),A.fd()),h=U.jpb(b[L],r),o=U.jpb(Q[L],r),o!=null&&(e=z(l.d7,S1,7,2,0,2),e[0]=h,e[1]=o,U.Uvb(D[L],r,e));return D;},a.CJ=function(t,e){var A,P,r,i,D;for(i=z(l.mdb,l1,44,8,0,1),D=0;D<=7;D++)for(i[D]=new U.bwb(new lt.E3()),A=0;A<t.f;A++)r=e[A][D],r!=null&&(P=U.jpb(i[D],r),P=P==null?z(l.d7,H,6,1,15,1):U.Eqb(P,P.length+1),P[P.length-1]=A,U.Uvb(i[D],r,P));return i;},a.DJ=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;for(D=z(l.d7,H,6,t.s,15,1),r=0;r<e.f;r++){for(b=e.j[r]+(e.i[r].length-e.e[r]),h=0,o=0;o<e.i[r].length;o++)if(o<e.j[r]||o>=e.e[r]){for(c=2*A[e.i[r][o]],i=e.k[r][o],n.em(e,i)==2&&(i<e.g&&n.ot(e.p,i)||++c),L=0;L<h&&!(c<D[L]);L++);for(s=h;s>L;s--)D[s]=D[s-1];D[L]=c,++h;}for(n.dh(t.b[r],r),n._g(t.b[r],t.a,l.Oeb(A[r])),Q=b;Q<t.s;Q++)n._g(t.b[r],t.a+1,0);for(u=0;u<b;u++)n._g(t.b[r],t.a+1,l.Oeb(D[u]));}for(P=e.f;P<t.b.length;P++)for(n.dh(t.b[P],P),n._g(t.b[P],t.a,Ct(e.f+1)),u=0;u<t.s;u++)n._g(t.b[P],t.a+1,0);},a.EJ=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(n.aw(t,7),o=new n.tw(t.f,t.g),u=h1(l._6,[bP,Te],[58,10],0,[t.f,8],2),P=z(l.d7,H,6,t.f,15,1),r=z(l.zeb,ht,6,t.f,16,1),i=z(l.zeb,ht,6,t.g,16,1),g=0;g<t.f;g++){for(g!=0&&(U.Sqb(r,r.length),U.Sqb(i,i.length)),u[g][0]=z(l._6,J,6,2,15,1),u[g][0][0]=t.C[g]<<24>>24,u[g][0][1]=t.A[g]<<24>>24,P[0]=g,r[g]=true,b=0,h=1,t.u[g]|=512,d=1;d<=7&&h<t.f;d++){for(c=h,Q=b;Q<h;Q++)for(A=P[Q],L=0;L<t.j[A];L++)D=t.i[A][L],r[D]||(r[D]=true,P[c++]=D),i[t.k[A][L]]=true;if(c==h)break;for(b=h,h=c,n.Co(t,o,i,true,null),e=0;e<o.q;e++)o.s[e]=0,o.T=0,n.vn(o,e,0);u[g][d]=w.xzb((s=n.qg(new n.Ng(o,16)),V1.Hmb(),s));}t.u[g]&=-513;}return u;},a.FJ=function(t,e){var A,P,r;for(A=0,U.Yqb(t.b),P=0;P<e.length;P++)(P==0||n.bh(t.b[P],t.b[P-1])!=0)&&++A,r=t.b[P].a,e[r]=A;return A;},a.GJ=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h;for(h=new U.Twb(new lt.V3()),Q=e,L=0,s=Q.length;L<s;++L)for(o=Q[L],r=A,i=0,D=r.length;i<D;++i)P=r[i],u=z(l.d7,H,6,2,15,1),u[0]=t.J[o],u[1]=t.C[P],U.Uvb(h.a,u,(T.Kgb(),kA))==null;return U.Li(h,z(l.d7,S1,7,0,0,2));},a.HJ=function(t,e,A){return n.Zo(t.u,A)!=0&&(t.H[t.G[e]]=true),n.Zo(t.D,e)!=0&&(t.A[t.w[A]]=true),new a.qJ(e,A);},a.IJ=function(t,e,A,P,r){var i,D,u,o,Q;for(i=A[P],Q=false,u=0;u<e.f;u++)if(u!=P&&A[u]==i){Q=true;break;}if(!Q)return r;for(D=0;D<e.f;D++)(D==P||A[D]>i)&&++A[D];++r;do o=r,a.DJ(t,e,A),r=a.FJ(t,A);while(o!=r);return r;},a.JJ=function(t){var e,A,P,r,i,D,u;for(D=z(l._6,bP,58,8,0,3),u=0;u<=7;u++)for(D[u]=z(l._6,Te,10,t[u].c,0,2),P=0,i=(A=new U.kwb(new U.qwb(new U.xpb(t[u]).a).b),new U.Gpb(A));U.fob(i.a.a);)r=(e=U.iwb(i.a),e.fd()),D[u][P++]=r;return D;},a.KJ=function(t,e){var A,P,r,i,D,u,o,Q,L;for(L=0,r=t,D=0,o=r.length;D<o;++D)A=r[D],e[A]==0&&++L;if(L==0)return null;for(Q=z(l.d7,H,6,L,15,1),L=0,P=t,i=0,u=P.length;i<u;++i)A=P[i],e[A]==0&&(Q[L++]=A);return Q;},a.LJ=function(t){return t.K++==64?false:t.i?a.xJ(t.i.b)?false:(a.RJ(t),true):(t.i=new a.uJ(),true);},a.MJ=function(t){var e,A,P,r,i;for(i=j.Math.max(t.D.f,t.u.f),t.a=n.Og(i),t.s=2,A=0;A<t.D.f;A++)t.s=j.Math.max(t.s,n.Zo(t.D,A)+n.qp(t.D,A));for(P=0;P<t.u.f;P++)t.s=j.Math.max(t.s,n.Zo(t.u,P)+n.qp(t.u,P));for(r=j.Math.max(2,(62+t.a+t.s*(t.a+5))/63|0),t.b=z(l.p7,Mn,101,i,0,1),e=0;e<i;e++)t.b[e]=new n.eh(r);t.F=new n.Ng(t.D,1),t.v=new n.Ng(t.u,1);},a.NJ=function(t){var e,A,P,r,i,D,u,o,Q;for(Q=t.F.g,t.J=U.Eqb(Q,Q.length),e=t.v.g,t.C=U.Eqb(e,e.length),r=t.J,D=0,o=r.length;D<o;++D)A=r[D],t.o<A&&(t.o=A);for(P=t.C,i=0,u=P.length;i<u;++i)A=P[i],t.n<A&&(t.n=A);},a.OJ=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g;if(g=a.KJ(e,t.I),g==null||(c=a.KJ(A,t.B),c==null))return null;if(g.length==1&&c.length==1)return new a.qJ(g[0],c[0]);if(b=a.GJ(t,g,c),b.length==1)return new a.qJ(g[0],c[0]);for(L=a.sJ(t.i,b.length),h=-1,i=g,u=0,Q=i.length;u<Q;++u)if(P=i[u],t.J[P]==b[L][0]){h=P;break;}for(s=-1,r=c,D=0,o=r.length;D<o;++D)if(P=r[D],t.C[P]==b[L][1]){s=P;break;}return t.o=a.IJ(t,t.D,t.J,h,t.o),t.n=a.IJ(t,t.u,t.C,s,t.n),new a.qJ(h,s);},a.PJ=function(t){var e,A,P,r,i;for(P=a.QJ(t);P;){for(i=false,A=0;A<n.Zo(t.D,P.b);A++)t.I[n.Yo(t.D,P.b,A)]==0&&(i=true);for(r=false,e=0;e<n.Zo(t.u,P.a);e++)t.B[n.Yo(t.u,P.a,e)]==0&&(r=true);if(i&&r)break;t.I[P.b]=cP,t.B[P.a]=cP,P=a.QJ(t);}if(!P){for(A=0;A<t.I.length;A++)t.I[A]==cP&&(t.I[A]=0);for(e=0;e<t.B.length;e++)t.B[e]==cP&&(t.B[e]=0);}return P;},a.QJ=function(t){for(var e,A,P,r,i;t.t.a.length!=0;)if(P=U.Dj(t.t,0),t.I[P.b]==0&&t.B[P.a]==0)return P;for(;t.g>=0;){for(;t.g>=2&&t.c<t.j[t.g].length;){if(A=t.j[t.g][t.c],e=U.jpb(t.k[t.g],A),n.Sl(t.D,e[0][0])==6&&(P=a.OJ(t,e[0],e[1]),P))return P;++t.c;}for(;t.g>=2&&t.d<t.j[t.g].length;){if(A=t.j[t.g][t.d],e=U.jpb(t.k[t.g],A),n.Sl(t.D,e[0][0])!=6&&(P=a.OJ(t,e[0],e[1]),P))return P;++t.d;}for(;t.p&&t.g==0&&t.e<t.j[0].length;)if(A=t.j[0][t.e++],e=U.jpb(t.k[t.g],A),e[0].length==1&&e[1].length==1&&(P=a.SJ(t,e[0][0],e[1][0]),P))return P;for(;t.g==0&&t.f<t.j[t.g].length;)if(A=t.j[t.g][t.f++],e=U.jpb(t.k[t.g],A),e[0].length>=e[1].length&&a.zJ(e[0],t.I,t.G,t.H,t.J)||e[1].length>=e[0].length&&a.zJ(e[1],t.B,t.w,t.A,t.C)){for(i=0,r=0;i<e[0].length&&t.I[e[0][i]]!=0;)++i;for(;r<e[1].length&&t.B[e[1][r]]!=0;)++r;if(i<e[0].length&&r<e[1].length&&(P=a.SJ(t,e[0][i],e[1][r]),P))return P;}--t.g,t.c=0,t.d=0,t.e=0,t.f=0;}return null;},a.RJ=function(t){U.Iqb(t.I,0),U.Iqb(t.B,0),U.Mqb(t.H),U.Mqb(t.A),a.NJ(t),a.tJ(t.i),t.q=a.AJ(t),t.g=7,t.c=0,t.d=0,t.e=0,t.f=0;},a.SJ=function(t,e,A){return t.L!=null&&t.L[e]!=null&&t.L[e][A]?null:t.I[e]==0&&t.B[A]==0?a.HJ(t,e,A):null;},a.TJ=function(t,e,A,P,r){var i,D,u,o,Q;for(this.D=t,this.u=e,this.I=A,this.B=P,this.L=r,a.MJ(this),a.NJ(this),this.k=a.BJ(this),this.j=a.JJ(this.k),this.r=0,Q=(u=new U.kwb(new U.qwb(new U.xpb(this.k[0]).a).b),new U.Gpb(u));U.fob(Q.a.a);)o=(D=U.iwb(Q.a),D.fd()),i=U.jpb(this.k[0],o),this.r+=j.Math.min(i[0].length,i[1].length);this.p=this.r==this.D.f&&this.D.f==this.u.f,this.G=z(l.d7,H,6,this.D.f,15,1),this.w=z(l.d7,H,6,this.u.f,15,1),this.H=z(l.zeb,ht,6,n.gp(this.D,this.G,false,false),16,1),this.A=z(l.zeb,ht,6,n.gp(this.u,this.w,false,false),16,1),this.q=a.AJ(this),this.g=7,this.K=0;},x(397,1,{},a.TJ),y.a=0,y.c=0,y.d=0,y.e=0,y.f=0,y.g=0,y.n=0,y.o=0,y.p=false,y.q=0,y.r=0,y.s=0,y.K=0,l.I9=X(397),a.UJ=function(t,e,A){var P,r,i,D,u;for(this.a=z(l.d7,H,6,t.f,15,1),this.a[0]=e,A[e]=true,i=0,D=0;i<=D;){for(r=n.Zo(t,this.a[i]),u=0;u<r;u++)P=n.Yo(t,this.a[i],u),j.Math.abs(t.w[P])==0&&M1(G(t.B[P],r0),0)&&!A[P]&&(this.a[++D]=P,A[P]=true);++i;}},x(395,1,{},a.UJ),l.K9=X(395),a.VJ=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M;for(n.aw(t,7),L=new n.tw(t.f,t.g),Q=h1(l._6,[c0,bP],[204,58],0,[t.f,8],2),P=z(l.d7,H,6,t.f,15,1),r=z(l.d7,H,6,t.f,15,1),i=z(l.zeb,ht,6,t.f,16,1),v=0;v<t.f;v++)if(Q[v]=h1(l._6,[bP,Te],[58,10],0,[t.j[v],8],2),!(e&&t.C[v]!=6))for(o=0;o<t.j[v];o++){for(s=t.i[v][o],U.Sqb(i,i.length),m=1,d=2,t.u[s]|=512,M=0;M<8&&d<t.f;M++){if(M==0)P[0]=s,i[s]=true,P[1]=v,i[v]=true;else {for(C=d,b=m;b<d;b++)for(A=P[b],c=0;c<t.j[A];c++)u=t.i[A][c],!i[u]&&(!e||t.C[u]==6)&&(i[u]=true,P[C++]=u);if(C==d){if(!e)for(h=M;h<8;h++)Q[v][o][h]=Q[v][o][h-1];break;}m=d,d=C;}if(M==0)Q[v][o][M]=z(l._6,J,6,2,15,1),Q[v][o][M][0]=t.C[v]<<24>>24,Q[v][o][M][1]=t.A[v]<<24>>24;else {for(n.Bo(t,L,i,true,r),n.hn(L,r[s],0),n.vn(L,r[s],0),A=0;A<t.f;A++)r[A]!=-1&&t.j[A]>n.Zo(L,r[A])&&n.un(L,r[A],Y1,true);if(e)for(D=0;D<L.g;D++)L.H[D]=1,L.T=0;Q[v][o][M]=w.xzb((g=n.qg(new n.Ng(L,16)),V1.Hmb(),g));}}t.u[s]&=-513;}return Q;},a.WJ=function(t,e,A){var P,r;for(P=0,r=0;r<t.j[e];r++)n.Sl(t,t.i[e][r])==A&&++P;return P;},a.XJ=function(t,e,A,P,r){var i,D,u,o,Q,L,s,h;for(h=0,s=-1,D=0;D<t.j.f;D++){L=U.wj(e.g,h),n.aw(L,1);do++s==L.f&&(s=0,L=tt.oH(e,++h));while(L.K&&kt(G(L.B[s],r0),0));n.qn(L,s,A[D],A[D]<=r);}for(Q=0,o=-1,i=0;i<t.d.f;i++){u=U.wj(e.f,Q),n.aw(u,1);do++o==u.f&&(o=0,u=tt.nH(e,++Q));while(u.K&&kt(G(u.B[o],r0),0));n.qn(u,o,P[i],P[i]<=r);}},a.YJ=function(t,e,A,P,r){var i,D,u,o,Q;for(Q=-1,D=0;D<n.Zo(t.j,A);D++)if(n.Yo(t.j,A,D)==e){Q=D;break;}for(u=-1,i=0;i<n.Zo(t.d,r);i++)if(n.Yo(t.d,r,i)==P){u=i;break;}for(o=0;o<8;o++)if(lt.D3(t.k[A][Q][o],t.e[r][u][o])!=0)return o<<8;return z1;},a.ZJ=function(t,e){return e<t.g&&(n.tt(t.p,e)||t.H[e]==8)?0:t.H[e]&127;},a.$J=function(t,e,A){var P,r;for(r=0;r<t.j[A];r++)if(P=t.i[A][r],P!=e)return P;return  -1;},a._J=function(t,e){var A,P;for(A=t.o[e],P=0;P<t.j[e];P++)n.Gm(t,t.i[e][P])&&++A;return A;},a.aK=function(t,e,A,P,r){var i,D,u,o,Q;for(Q=-1,D=0;D<n.Zo(t.j,A);D++)if(n.Yo(t.j,A,D)==e){Q=D;break;}for(u=-1,i=0;i<n.Zo(t.d,r);i++)if(n.Yo(t.d,r,i)==P){u=i;break;}for(o=2;o<8;o++)if(t.p[A][Q][o]==null||lt.D3(t.p[A][Q][o],t.i[r][u][o])!=0)return o-2<<8;return 1536;},a.bK=function(t){var e,A,P,r,i,D,u,o;for(u=z(l.d7,H,6,t.f,15,1),o=(n.aw(t,7),t.p),i=0;i<j.Math.min(32,o.j.a.length);i++)for(D=U.wj(o.j,i),A=D,P=0,r=A.length;P<r;++P)e=A[P],u[e]|=1<<i;return u;},a.cK=function(t,e,A,P){return (P[t]&P[e]&~P[A])!=0;},a.dK=function(t,e,A,P,r,i){var D,u,o,Q,L,s,h,b;for(t.j=e,t.d=A,t.n=z(l.d7,H,6,P.length,15,1),t.f=z(l.d7,H,6,r.length,15,1),t.k=a.VJ(t.j,false),t.p=a.VJ(t.j,true),t.e=a.VJ(t.d,false),t.i=a.VJ(t.d,true),t.o=a.bK(t.j),t.g=a.bK(t.d),t.q=Er,s=new a.TJ(e,A,t.n,t.f,i),t.a=0;a.LJ(s);){for(++t.a,t.c=s.q,Q=s.r,L=a.PJ(s);L;)a.eK(t,L),L=a.PJ(s);if(t.b=t.c,t.c<Q?(D=new a.cJ(t.j,t.d,t.n,t.f,t.c,i),h=a.bJ(D),t.c+=D.b-D.g):(b=new a.aJ(t.j,t.d),h=a._I(b,a.UI(b,t.n,t.f))),t.q<h){for(t.q=h,o=0;o<P.length;o++)P[o]=t.n[o];for(u=0;u<r.length;u++)r[u]=t.f[u];}}},a.eK=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K;for(Q=z(l.d7,H,6,t.j.f,15,1),m=z(l.d7,H,6,t.d.f,15,1),L=z(l.d7,H,6,t.j.f,15,1),v=z(l.d7,H,6,t.d.f,15,1),Q[0]=e.b,L[e.b]=e.b,m[0]=e.a,v[e.a]=e.a,t.n[e.b]==0&&(++t.c,t.n[e.b]=t.c,t.f[e.a]=t.c),u=0,s=0;u<=s;){for(V=Q[u],M=m[u],d=h1(l.d7,[S1,H],[7,6],15,[n.Zo(t.j,V),n.Zo(t.d,M)],2),b=0;b<n.Zo(t.j,V);b++)if(S=n.Yo(t.j,V,b),t.n[S]==0)for(i=a.ZJ(t.j,n.$o(t.j,V,b)),g=0;g<n.Zo(t.d,M);g++)C=n.Yo(t.d,M,g),t.f[C]==0&&n.Sl(t.j,S)==n.Sl(t.d,C)&&(D=n.$o(t.d,M,g),K=a.aK(t,V,S,M,C),i==a.ZJ(t.d,D)||K!=0?a.hK(t,V,S,M,C)&&(o=a.YJ(t,V,S,M,C),O=j.Math.max(K,o),O!=0&&(c=a.fK(t,L[V],V,S,v[M],M,C),a.iK(t,V,S,M,C,O,c)&&(d[b][g]=O,c&&(d[b][g]+=64),a.cK(L[V],V,S,t.o)==a.cK(v[M],M,C,t.g)&&(d[b][g]+=128),a._J(t.j,S)==a._J(t.d,C)&&(d[b][g]+=1)))):a.YJ(t,V,S,M,C)==z1&&(d[b][g]=768));for(;;){for(A=0,r=-1,P=-1,h=0;h<d.length;h++)if(S=n.Yo(t.j,V,h),t.n[S]==0)for(g=0;g<d[h].length;g++)C=n.Yo(t.d,M,g),t.f[C]==0&&A<d[h][g]&&(A=d[h][g],r=S,P=C);if(A==0)break;++s,Q[s]=r,L[r]=Q[u],m[s]=P,v[P]=m[u],++t.c,t.n[r]=t.c,t.f[P]=t.c;}++u;}return s+1;},a.fK=function(t,e,A,P,r,i,D){var u,o,Q,L,s;if(n.Zo(t.j,A)==3&&(n.Ml(t.j,A)==1||n.Ml(t.j,A)==2)&&n.Zo(t.d,i)==3&&(n.Ml(t.d,i)==1||n.Ml(t.d,i)==2)){for(s=e>P,Q=-1,o=0;o<n.Zo(t.j,A);o++)if(Q=n.Yo(t.j,A,o),Q!=e&&Q!=P){(Q>P&&Q<e||Q<P&&Q>e)&&(s=!s);break;}for(L=r>D,u=0;u<n.Zo(t.d,i);u++)if(Q=n.Yo(t.d,i,u),Q!=r&&Q!=D){(Q>D&&Q<r||Q<D&&Q>r)&&(L=!L);break;}return s==L==(n.Ml(t.j,A)==n.Ml(t.d,i));}return  true;},a.gK=function(t,e){var A,P,r,i;for(t.j=new n.sw(),P=0;P<e.g.a.length;P++)i=U.wj(e.g,P),n.aw(i,1),n.Yk(t.j,i,i.f,i.g,!i.K);for(t.d=new n.sw(),A=0;A<e.f.a.length;A++)r=U.wj(e.f,A),n.aw(r,1),n.Yk(t.d,r,r.f,r.g,!r.K);n.kq(t.j),n.kq(t.d);},a.hK=function(t,e,A,P,r){var i,D;return !(n.Sl(t.j,A)==6&&j.Math.abs(n.Zo(t.j,A)-n.Zo(t.d,r))>1||n.Sl(t.j,A)==8&&n.Zo(t.j,A)==2&&a.WJ(t.j,e,8)>a.WJ(t.d,P,8)||n.Sl(t.d,r)==8&&n.Zo(t.d,r)==2&&a.WJ(t.j,e,8)<a.WJ(t.d,P,8)||n.Sl(t.j,A)==8&&n.Zo(t.j,A)==2&&n.Zo(t.d,r)==2&&(D=n.Yo(t.j,A,0)==e?0:1,i=n.Yo(t.d,r,0)==P?0:1,lt.D3(t.p[A][D][3],t.i[r][i][3])!=0)||n.Sl(t.j,A)==7&&a.WJ(t.j,e,7)<a.WJ(t.d,P,7)||n.Sl(t.d,r)==7&&a.WJ(t.j,e,7)>a.WJ(t.d,P,7)||n.Xo(t.j,n.Wo(t.j,e,A))==3^n.Xo(t.d,n.Wo(t.d,P,r))==3);},a.iK=function(t,e,A,P,r,i,D){return !(n.Sl(t.j,A)==8&&n.Po(t.j,A)==0&&a.mK(t.j,e)&&a.mK(t.d,P)&&i!=z1||n.Sl(t.j,A)==8&&n.Zo(t.j,A)==2&&a.mK(t.j,a.$J(t.j,e,A))&&i<768||n.Sl(t.j,e)==8&&(a.mK(t.j,A)||a.mK(t.d,r))&&i<512||!D&&(n.Sl(t.j,A)!=6||!a.kK(t.j,e))||n.Sl(t.j,A)==8&&n.Sl(t.j,e)!=6&&i!=z1||n.Sl(t.j,e)==5&&n.Sl(t.j,A)==6&&i<768);},a.jK=function(){},a.kK=function(t,e){var A;for(A=0;A<t.j[e];A++)if(n.Sl(t,t.i[e][A])!=6)return  true;return  false;},a.lK=function(t,e,A){var P;for(P=0;P<t.j[e];P++)if(n.Sl(t,t.i[e][P])!=6&&t.i[e][P]!=A)return  true;return  false;},a.mK=function(t,e){var A;for(A=0;A<t.j[e];A++)if(t.n[e][A]==2&&n.Sl(t,t.i[e][A])>6)return  true;return  false;},a.nK=function(t,e,A){var P;for(P=0;P<t.j[e];P++)if(t.n[e][P]==2&&n.Sl(t,t.i[e][P])>6&&t.i[e][P]!=A)return  true;return  false;},x(394,1,{},a.jK),y.a=0,y.b=0,y.c=0,y.q=0,l.L9=X(394),q.rK=function(){q.rK=o1,q.pK=F(k(l.Sbb,1),st,2,6,["","r0","r","r3","r3-4","r3-5","r3-6","r3-7","r4","r4-5","r4-6","r4-7","r>3","r5","r5-6","r5-7","r>4","r6","r6-7","r>5","r7","r>6","r>7"]),q.oK=F(k(l.Sbb,1),st,2,6,[ce,dD,"is in any ring","3 members","3-4 members","3-5 members","3-6 members","3-7 members","4 members","4-5 members","4-6 members","4-7 members","> 3 members","5 members","5-6 members","5-7 members","> 4 members","6 members","6-7 members","> 5 members","7 members","> 6 members","> 7 members"]),q.qK=F(k(l.e7,1),n0,6,14,[0,Qe,K8,Se,25769803776,60129542144,128849018880,266287972352,qe,51539607552,120259084288,257698037760,532575944704,Ve,103079215104,240518168576,515396075520,Ee,206158430208,481036337152,Ne,412316860416,un]);},q.sK=function(t,e,A,P){var r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S;for(t.A=e,t.a=A,s=Ft(j.Math.round((j1.IN==-1&&(j1.IN=j1.KN()),j1.IN*8))),h=F(k(l.d7,1),H,6,15,[s,s/2|0,s*3/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s/2|0,s*3/2|0,s/4|0,s/2|0,s/2|0,s/4|0]),b=F(k(l.d7,1),H,6,15,[s*3/2|0,s/2|0]),c=F(k(l.d7,1),H,6,15,[s,-2,s,-2,s]),S=z(l.d7,H,6,1+2*h.length+(P?2*b.length:0),15,1),d=0,D=h,o=0,L=D.length;o<L;++o)r=D[o],S[d++]=r,S[d++]=-2;if(P)for(i=b,u=0,Q=i.length;u<Q;++u)r=i[u],S[d++]=r,S[d++]=-2;for(S[d++]=s,R.u2(t.v,c,S),t.b=R.o2(t.v,"Any atomic number"),R.Q1(t.b,t),R.n2(t.v,t.b,1,1,3,1),t.w=R.q2(t.v,Nr),t.D=R.r2(t.v,16,1),R.m2(t.v,t.w,1,3),R.m2(t.v,t.D,3,3),t.g=R.p2(t.v),R.e2(t.g,ce),R.e2(t.g,"is aromatic"),R.e2(t.g,"is hetero-aromatic"),R.e2(t.g,"is not aromatic"),R.m2(t.v,R.q2(t.v,"Aromaticity:"),1,5),R.m2(t.v,t.g,3,5),t.s=R.p2(t.v),R.e2(t.s,ce),R.e2(t.s,"0 (not in a ring)"),R.e2(t.s,"0 or 2 (0 or 1 ring)"),R.e2(t.s,">=2 (any ring count)"),R.e2(t.s,"2 (in 1 ring)"),R.e2(t.s,"3 (bridge head; 2 rings)"),R.e2(t.s,">3 (in more than 2 rings)"),R.m2(t.v,R.q2(t.v,"Ring bonds:"),1,7),R.m2(t.v,t.s,3,7),t.t=R.p2(t.v),R.e2(t.t,ce),R.e2(t.t,"3 members"),R.e2(t.t,"4 members"),R.e2(t.t,"5 members"),R.e2(t.t,"6 members"),R.e2(t.t,"7 members"),R.m2(t.v,R.q2(t.v,"Smallest ring size:"),1,9),R.m2(t.v,t.t,3,9),t.r=R.p2(t.v),C=q.oK,v=0,M=C.length;v<M;++v)m=C[v],R.e2(t.r,m);for(R.m2(t.v,R.q2(t.v,"Any ring size:"),1,11),R.m2(t.v,t.r,3,11),t.i=R.p2(t.v),R.e2(t.i,ce),R.e2(t.i,"not charged"),R.e2(t.i,"has negative charge"),R.e2(t.i,"has positive charge"),R.m2(t.v,R.q2(t.v,"Charge:"),1,13),R.m2(t.v,t.i,3,13),t.n=R.p2(t.v),R.e2(t.n,ce),R.e2(t.n,wP),R.e2(t.n,gP),R.e2(t.n,"exactly 3"),R.e2(t.n,Ir),R.e2(t.n,jD),R.e2(t.n,zP),R.e2(t.n,xr),R.e2(t.n,Or),R.e2(t.n,BD),R.e2(t.n,Xr),R.e2(t.n,"1,2, or 3"),R.e2(t.n,"2 or 3"),R.m2(t.v,R.q2(t.v,"Non-H neighbours:"),1,15),R.m2(t.v,t.n,3,15),t.j=R.p2(t.v),R.e2(t.j,ce),R.e2(t.j,"exactly 0"),R.e2(t.j,wP),R.e2(t.j,gP),R.e2(t.j,"exactly 3"),R.e2(t.j,mD),R.e2(t.j,Ir),R.e2(t.j,jD),R.e2(t.j,zP),R.e2(t.j,xr),R.e2(t.j,Or),R.e2(t.j,BD),R.e2(t.j,Xr),R.e2(t.j,"1,2, or 3"),R.e2(t.j,"2 or 3"),R.m2(t.v,R.q2(t.v,"Electroneg. neighbours:"),1,17),R.m2(t.v,t.j,3,17),t.k=R.p2(t.v),R.e2(t.k,ce),R.e2(t.k,"none"),R.e2(t.k,wP),R.e2(t.k,gP),R.e2(t.k,zP),R.e2(t.k,xr),R.e2(t.k,Or),R.e2(t.k,mD),R.e2(t.k,Ir),R.e2(t.k,Xr),R.m2(t.v,R.q2(t.v,"Hydrogen count:"),1,19),R.m2(t.v,t.k,3,19),t.p=R.p2(t.v),R.e2(t.p,ce),R.e2(t.p,"none"),R.e2(t.p,wP),R.e2(t.p,gP),R.e2(t.p,zP),R.m2(t.v,R.q2(t.v,"Pi-electron count:"),1,21),R.m2(t.v,t.p,3,21),t.o=R.p2(t.v),R.e2(t.o,ce),g=1;g<15;g++)R.e2(t.o,""+(g-7));R.m2(t.v,R.q2(t.v,"Oxidation state:"),1,23),R.m2(t.v,t.o,3,23),t.c=R.o2(t.v,"prohibit further substitution"),R.Q1(t.c,t),R.n2(t.v,t.c,1,25,3,25),t.f=R.o2(t.v,"require further substitution"),R.Q1(t.f,t),R.n2(t.v,t.f,1,27,3,27),t.u=R.p2(t.v),R.e2(t.u,ce),R.e2(t.u,"is a stereo center"),R.e2(t.u,"is not a stereo center"),R.m2(t.v,R.q2(t.v,"Stereo center:"),1,29),R.m2(t.v,t.u,3,29),t.e=R.o2(t.v,"match stereo center"),R.n2(t.v,t.e,1,31,3,31),t.d=R.o2(t.v,"is part of exclude group"),R.n2(t.v,t.d,1,33,3,33),P&&(R.n2(t.v,R.q2(t.v,"Stereo center hint for product:"),1,35,3,35),t.q=R.p2(t.v),R.e2(t.q,"Copy from generic product"),R.e2(t.q,"Keep reactant configuration"),R.e2(t.q,"Invert reactant configuration"),R.e2(t.q,"Racemise configuration"),R.n2(t.v,t.q,1,37,3,37)),n.aw(t.A,31),q.vK(t);},q.tK=function(t){var e,A,P,r,i,D,u,o,Q;for(u=null,o=T.wjb(R.m3(t.D));w.Xzb(o).length!=0;){if(A=T.ljb(o,Z1(44)),A==-1?(D=T.wjb(o),o=""):(D=T.wjb((w.Rzb(0,A,w.Xzb(o).length),w.Xzb(o).substr(0,A))),o=T.wjb((w.Szb(A+1,w.Xzb(o).length+1),w.Xzb(o).substr(A+1)))),T.ejb(D,"X")){w.Xzb(o).length!=0&&(o=","),o=(w.Lzb(o),o+(w.Lzb("F,Cl,Br,I"),"F,Cl,Br,I"));continue;}if(e=(n.Qk(),n.oo(D,321)),e!=0)if(e==1)R.w2(t.v,"'H' cannot be part of an atom list and is removed.");else if(u==null)u=z(l.d7,H,6,1,15,1),u[0]=e;else {for(P=false,i=0;i<u.length;i++)if(e==u[i]){P=true;break;}if(!P){for(Q=z(l.d7,H,6,u.length+1,15,1),r=0;r<u.length;r++)Q[r]=u[r];Q[u.length]=e,u=Q;}}}return u!=null&&w.qzb(u,w.Xzb(h0(U.qrb.prototype.jd,U.qrb,[]))),u;},q.uK=function(t,e){e.j==0?(q.wK(t),t.B=true,R.s2(t.v)):e.j==1?R.s2(t.v):k1(e.k)===k1(t.b)?e.a==1?(R.n3(t.D,""),R.f3(t.w,Nr)):(R.n3(t.D,n.Hl(t.A,t.a)),R.f3(t.w,kD)):k1(e.k)===k1(t.c)?(R.Y1(t.f,false),R.j2(t.n,0),R.j2(t.j,0)):k1(e.k)===k1(t.f)&&R.Y1(t.c,false);},q.vK=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d;for(L=n.Nl(t.A,t.a),kt(G(L,1),0)?(R.Y1(t.b,true),R.f3(t.w,Nr)):R.f3(t.w,kD),R.n3(t.D,n.Il(t.A,t.a)==null?"":n.Jl(t.A,t.a)),e=G(L,ZA),kt(G(e,AA),0)?R.j2(t.g,2):et(e,2)==0?R.j2(t.g,1):et(e,4)==0?R.j2(t.g,3):R.j2(t.g,0),b=G(L,120),et(b,112)==0?R.j2(t.s,1):et(b,96)==0?R.j2(t.s,2):et(b,8)==0?R.j2(t.s,3):et(b,104)==0?R.j2(t.s,4):et(b,88)==0?R.j2(t.s,5):et(b,56)==0?R.j2(t.s,6):R.j2(t.s,0),g=Gt(t0(G(L,Be),22)),R.j2(t.t,g==0?0:g-2),h=G(L,Me),u=-1,D=0;D<q.qK.length;D++)if(M1(h,q.qK[D])){u=D;break;}u!=-1?R.j2(t.r,u):(P=new T.Rjb("Custom:"),kt(G(h,Qe),0)&&(P.a+=" 0"),kt(G(h,Se),0)&&(P.a+=" 3"),kt(G(h,qe),0)&&(P.a+=" 4"),kt(G(h,Ve),0)&&(P.a+=" 5"),kt(G(h,Ee),0)&&(P.a+=" 6"),kt(G(h,Ne),0)&&(P.a+=" 7"),kt(G(h,un),0)&&(P.a+=" >=8"),t.C=h,R.e2(t.r,P.a),R.j2(t.r,q.qK.length)),o=G(L,ae),et(o,rA)==0?R.j2(t.n,1):et(o,pn)==0?R.j2(t.n,2):et(o,Rn)==0?R.j2(t.n,3):et(o,iA)==0?R.j2(t.n,4):et(o,F0)==0?R.j2(t.n,5):et(o,A0)==0?R.j2(t.n,6):et(o,tn)==0?R.j2(t.n,7):et(o,ie)==0?R.j2(t.n,8):et(o,Qn)==0?R.j2(t.n,9):et(o,3276800)==0?R.j2(t.n,10):et(o,2228224)==0?R.j2(t.n,11):et(o,2490368)==0?R.j2(t.n,12):R.j2(t.n,0),r=G(L,Ln),et(r,NA)==0?R.j2(t.j,1):et(r,IA)==0?R.j2(t.j,2):et(r,xA)==0?R.j2(t.j,3):et(r,OA)==0?R.j2(t.j,4):et(r,tr)==0?R.j2(t.j,5):et(r,er)==0?R.j2(t.j,6):et(r,fA)==0?R.j2(t.j,7):et(r,Cn)==0?R.j2(t.j,8):et(r,XA)==0?R.j2(t.j,9):et(r,GA)==0?R.j2(t.j,10):et(r,DA)==0?R.j2(t.j,11):et(r,nr)==0?R.j2(t.j,12):et(r,Ar)==0?R.j2(t.j,13):et(r,Pr)==0?R.j2(t.j,14):R.j2(t.j,0),A=G(L,je),et(A,PA)==0?R.j2(t.i,1):et(A,ln)==0?R.j2(t.i,2):et(A,on)==0?R.j2(t.i,3):R.j2(t.i,0),i=G(L,Z0),et(i,1792)==0?R.j2(t.k,1):et(i,1664)==0?R.j2(t.k,2):et(i,1408)==0?R.j2(t.k,3):et(i,128)==0?R.j2(t.k,4):et(i,384)==0?R.j2(t.k,5):et(i,896)==0?R.j2(t.k,6):et(i,1536)==0?R.j2(t.k,7):et(i,i0)==0?R.j2(t.k,8):et(i,1152)==0?R.j2(t.k,9):R.j2(t.k,0),Q=G(L,$e),et(Q,98304)==0?R.j2(t.p,1):et(Q,81920)==0?R.j2(t.p,2):et(Q,49152)==0?R.j2(t.p,3):et(Q,x0)==0?R.j2(t.p,4):R.j2(t.p,0),s=t0(G(L,ur),47),R.j2(t.o,Gt(s)),kt(G(L,z1),0)&&R.Y1(t.c,true),kt(G(L,Y1),0)&&R.Y1(t.f,true),d=G(L,Dr),M1(d,eA)?R.j2(t.u,1):M1(d,nA)?R.j2(t.u,2):R.j2(t.u,0),kt(G(L,Q0),0)&&R.Y1(t.e,true),kt(G(L,r0),0)&&R.Y1(t.d,true),t.q&&(c=G(L,he),et(c,E0)==0?R.j2(t.q,1):et(c,dA)==0?R.j2(t.q,2):et(c,he)==0?R.j2(t.q,3):R.j2(t.q,0));},q.wK=function(t){var e,A;if(A=q.tK(t),n.Om(t.A,t.a))for(e=0;e<t.A.q;e++)n.Om(t.A,e)&&q.xK(t,e,A);else q.xK(t,t.a,A);n.Bq(t.A);},q.xK=function(t,e,A){var P,r,i,D,u,o;for(i=0,R.X1(t.b)?(i=Pt(i,1),n.pn(t.A,e,A,true)):n.pn(t.A,e,A,false),R.f2(t.g)==2?n._p(t.A,e)||(i=Pt(i,yr)):n.Sp(t.A,e)||(R.f2(t.g)==1?i=Pt(i,2):R.f2(t.g)==3&&(i=Pt(i,4))),u=0,r=0;r<n.Zo(t.A,e);r++)kt(G(n.Nl(t.A,n.Yo(t.A,e,r)),r0),0)&&n.fq(t.A,n.$o(t.A,e,r))&&++u;switch(R.f2(t.s)){case 1:u==0&&(i=Pt(i,112));break;case 2:u<=2&&(i=Pt(i,96));break;case 3:i=Pt(i,8);break;case 4:u<3&&(i=Pt(i,104));break;case 5:u<4&&(i=Pt(i,88));break;case 6:i=Pt(i,56);}switch(R.f2(t.t)!=0&&(i=Pt(i,b1(Ct(R.f2(t.t)+2),22))),o=R.f2(t.r),o==q.qK.length?i=Pt(i,t.C):o!=0&&(i=Pt(i,q.qK[o])),R.f2(t.i)){case 1:i=Pt(i,PA);break;case 2:i=Pt(i,ln);break;case 3:i=Pt(i,on);}switch(D=n.tp(t.A,e),R.f2(t.n)){case 1:D==1?i=Pt(i,z1):D<1&&(i=Pt(i,rA));break;case 2:D==2?i=Pt(i,z1):D<2&&(i=Pt(i,pn));break;case 3:D==3?i=Pt(i,z1):D<3&&(i=Pt(i,Rn));break;case 4:D==2?i=Pt(i,z1):D<2&&(i=Pt(i,iA));break;case 5:D==3?i=Pt(i,z1):D<3&&(i=Pt(i,F0));break;case 6:D==0&&(i=Pt(i,Y1));break;case 7:D==1?i=Pt(i,Y1):D<1&&(i=Pt(i,tn));break;case 8:D==2?i=Pt(i,Y1):D<2&&(i=Pt(i,ie));break;case 9:D==3?i=Pt(i,Y1):D<3&&(i=Pt(i,Qn));break;case 10:D==0?i=Pt(i,3276800):i=Pt(i,iA);break;case 11:D==0?i=Pt(i,2228224):i=Pt(i,F0);break;case 12:D<=1?i=Pt(i,1099513856e3):D<=3&&(i=Pt(i,F0));}switch(P=n.Oo(t.A,e),R.f2(t.j)){case 1:P==0&&(i=Pt(i,NA));break;case 2:P<=1&&(i=Pt(i,IA));break;case 3:P<=2&&(i=Pt(i,xA));break;case 4:P<=3&&(i=Pt(i,OA));break;case 5:P<2&&(i=Pt(i,tr));break;case 6:P<3&&(i=Pt(i,er));break;case 7:P<4&&(i=Pt(i,fA));break;case 8:P==0&&(i=Pt(i,Cn));break;case 9:P<2&&(i=Pt(i,XA));break;case 10:P<3&&(i=Pt(i,GA));break;case 11:P<4&&(i=Pt(i,DA));break;case 12:P<2&&(i=Pt(i,nr));break;case 13:P<3&&(i=Pt(i,Ar));break;case 14:P<3&&(i=Pt(i,Pr));}switch(R.f2(t.k)){case 1:i=Pt(i,1792);break;case 2:i=Pt(i,1664);break;case 3:i=Pt(i,1408);break;case 4:i=Pt(i,128);break;case 5:i=Pt(i,384);break;case 6:i=Pt(i,896);break;case 7:i=Pt(i,1536);break;case 8:i=Pt(i,i0);break;case 9:i=Pt(i,1152);}switch(R.f2(t.p)){case 1:i=Pt(i,98304);break;case 2:i=Pt(i,81920);break;case 3:i=Pt(i,49152);break;case 4:i=Pt(i,x0);}switch(R.f2(t.o)!=0&&(i=Pt(i,b1(Ct(R.f2(t.o)),47))),R.X1(t.c)&&(n.ip(t.A,e)>0||n.Al(t.A,e)==0&&(n.Sl(t.A,e)==5||n.aq(t.A,e)||n.Wp(t.A,e)))&&(i=Pt(i,z1)),R.X1(t.f)&&(n.ip(t.A,e)>0||n.Al(t.A,e)==0&&(n.Sl(t.A,e)==5||n.aq(t.A,e)||n.Wp(t.A,e)))&&(i=Pt(i,Y1)),R.f2(t.u)){case 1:i=Pt(i,eA);break;case 2:i=Pt(i,nA);}if(R.X1(t.e)&&(i=Pt(i,Q0)),R.X1(t.d)&&(i=Pt(i,r0)),t.q)switch(R.f2(t.q)){case 1:i=Pt(i,E0);break;case 2:i=Pt(i,dA);break;case 3:i=Pt(i,he);}n.un(t.A,e,-1,false),n.un(t.A,e,i,true);},q.yK=function(t){return t.B=false,R.v2(t.v),t.B;},q.zK=function(t,e,A,P){q.rK(),this.v=W.p1(t,(e.u[A]&512)!=0?"Atom Query Features (Multiple)":"Atom Query Features",this),q.sK(this,e,A,P);},x(292,1,hn,q.zK),y.lc=function(t){q.uK(this,t);},y.a=0,y.B=false,y.C=0,l.M9=X(292),q.BK=function(t,e,A){var P,r,i,D;for(t.u=e,t.a=A,P=Ft(j.Math.round((j1.IN==-1&&(j1.IN=j1.KN()),j1.IN*8))),r=F(k(l.d7,1),H,6,15,[P,-1,-2,-2,P]),D=F(k(l.d7,1),H,6,15,[P,-2,P,-2,-2,-2,-2,-2,-2,-2,P,-2,P,-2,P,-2,P,-2,2*P,-2,P/2|0,-2,P/2|0,-2,2*P]),R.u2(t.s,r,D),R.n2(t.s,R.q2(t.s,"Desired Bond type(s):"),1,1,3,1),t.k=R.o2(t.s,"Single"),R.n2(t.s,t.k,1,3,3,3),t.c=R.o2(t.s,"Double"),R.n2(t.s,t.c,1,4,3,4),t.n=R.o2(t.s,"Triple"),R.n2(t.s,t.n,1,5,3,5),t.i=R.o2(t.s,"Quadruple"),R.n2(t.s,t.i,1,6,3,6),t.j=R.o2(t.s,"Quintuple"),R.n2(t.s,t.j,1,7,3,7),t.b=R.o2(t.s,"Delocalized"),R.n2(t.s,t.b,1,8,3,8),t.g=R.o2(t.s,"Coordinate (0-order)"),R.n2(t.s,t.g,1,9,3,9),t.q=R.p2(t.s),R.e2(t.q,"any ring state"),R.e2(t.q,dD),R.e2(t.q,"is any ring bond"),R.e2(t.q,"is non-aromatic ring bond"),R.e2(t.q,"is aromatic bond"),R.e2(t.q,"is any non-aromatic bond"),R.Q1(t.q,t),R.n2(t.s,t.q,1,11,3,11),t.r=R.p2(t.s),R.e2(t.r,"any ring size"),R.e2(t.r,"is in 3-membered ring"),R.e2(t.r,"is in 4-membered ring"),R.e2(t.r,"is in 5-membered ring"),R.e2(t.r,"is in 6-membered ring"),R.e2(t.r,"is in 7-membered ring"),R.e2(t.r,"smallest ring 8 to 11"),R.e2(t.r,"smallest ring >= 12"),R.n2(t.s,t.r,1,13,3,13),t.e=R.o2(t.s,"Match formal bond order"),R.Y1(t.e,(e.G[A]&yn)!=0),R.Q1(t.e,t),R.n2(t.s,t.e,1,15,3,15),t.f=R.o2(t.s,"Match Stereo Configuration"),R.Y1(t.f,(e.G[A]&se)!=0),R.Q1(t.f,t),R.n2(t.s,t.f,1,17,3,17),t.d=R.o2(t.s,"Is atom bridge between"),R.Q1(t.d,t),R.n2(t.s,t.d,1,19,3,19),t.p=R.p2(t.s),i=0;i<16;i++)R.e2(t.p,""+i);R.m2(t.s,t.p,2,21),R.m2(t.s,R.q2(t.s," and"),3,21),R.Q1(t.p,t),t.o=R.p2(t.s),q.FK(t,0),R.m2(t.s,t.o,2,23),R.m2(t.s,R.q2(t.s," atoms"),3,23),n.aw(t.u,7),q.GK(t);},q.CK=function(t){var e;e=R.X1(t.d),R.S1(t.k,!e),R.S1(t.c,!e),R.S1(t.n,!e),R.S1(t.i,!e),R.S1(t.j,!e),R.S1(t.b,!e),R.S1(t.g,!e),R.S1(t.f,!e&&n.em(t.u,t.a)==2&&n.fm(t.u,t.a)!=0&&n.fm(t.u,t.a)!=3),R.S1(t.q,!e),R.S1(t.r,!e&&R.f2(t.q)!=1),R.S1(t.p,e),R.S1(t.o,e);},q.DK=function(t,e){var A,P,r;e.j==1?R.s2(t.s):e.j==0?(q.HK(t),t.v=true,R.s2(t.s)):k1(e.k)===k1(t.d)||k1(e.k)===k1(t.q)?q.CK(t):k1(e.k)===k1(t.p)&&(r=R.f2(t.p),t.t!=r&&(P=t.t+R.f2(t.o),A=q.FK(t,r),P<r?R.j2(t.o,0):P<r+A?R.j2(t.o,P-r):R.j2(t.o,A-1),t.t=r));},q.EK=function(t,e){return n.Om(t.u,n.Yl(t.u,0,e))&&n.Om(t.u,n.Yl(t.u,1,e));},q.FK=function(t,e){var A;for(R.h2(t.o),A=0;A<16;A++)R.e2(t.o,""+(e+A));return 16;},q.GK=function(t){var e,A,P,r,i,D,u;i=n.gm(t.u,t.a),P=n.hm(t.u,t.a)==8||n.Xp(t.u,t.a)?6:n.em(t.u,t.a),((i&1)!=0||P==1)&&R.Y1(t.k,true),((i&2)!=0||P==2)&&R.Y1(t.c,true),((i&4)!=0||P==3)&&R.Y1(t.n,true),((i&32)!=0||P==4)&&R.Y1(t.i,true),((i&64)!=0||P==5)&&R.Y1(t.j,true),((i&8)!=0||P==6)&&R.Y1(t.b,true),((i&16)!=0||P==0)&&R.Y1(t.g,true),(i&yn)!=0&&R.Y1(t.e,true),(i&se)!=0&&R.Y1(t.f,true),u=i&384,e=i&oA,u==128?R.j2(t.q,1):e==F0?R.j2(t.q,4):u==256?R.j2(t.q,e==0?2:3):R.j2(t.q,e==0?0:5),D=(i&ie)>>17,R.j2(t.r,D==0?0:D<=2?D+5:D-2),(i&Le)!=0?(R.Y1(t.d,true),r=(i&7680)>>9,A=(i&$A)>>13,R.j2(t.p,r),q.FK(t,r),R.j2(t.o,A)):R.j2(t.o,0),q.CK(t);},q.HK=function(t){var e;if(q.EK(t,t.a))for(e=0;e<t.u.r;e++)n.Om(t.u,n.Yl(t.u,0,e))&&n.Om(t.u,n.Yl(t.u,1,e))&&q.IK(t,e);else q.IK(t,t.a);n.Cq(t.u);},q.IK=function(t,e){var A,P,r,i,D;i=0,R.X1(t.d)?(r=R.f2(t.p),A=R.f2(t.o),i|=r<<9,i|=A<<13,i&=-128,n.Ln(t.u,e,1)):(R.X1(t.k)&&(i|=1),R.X1(t.c)&&(i|=2),R.X1(t.n)&&(i|=4),R.X1(t.i)&&(i|=32),R.X1(t.j)&&(i|=64),R.X1(t.b)&&(i|=8),R.X1(t.g)&&(i|=16),R.f2(t.q)!=0&&(R.f2(t.q)==1?n.fq(t.u,e)||(i|=128):R.f2(t.q)==2?n.fq(t.u,e)||(i|=256):R.f2(t.q)==3?n.Tp(t.u,e)||(i|=4194560):R.f2(t.q)==4?n.Tp(t.u,e)||(i|=F0):R.f2(t.q)==5&&(n.Tp(t.u,e)||(i|=_0))),R.f2(t.r)!=0&&(D=R.f2(t.r)+2,D>7&&(D-=7),P=n.Xo(t.u,e),(D<=2||D!=P)&&(i|=D<<17)),R.X1(t.e)&&(i|=yn),R.X1(t.f)&&(i|=se)),n.Kn(t.u,e,We,false),n.Kn(t.u,e,i,true),n.Rk(t.u,e);},q.JK=function(t){return t.v=false,R.v2(t.s),t.v;},q.KK=function(t,e,A){this.s=W.p1(t,n.Om(e,e.D[0][A])&&n.Om(e,e.D[1][A])?"Bond Query Features (Multiple)":"Bond Query Features",this),q.BK(this,e,A);},x(293,1,hn,q.KK),y.lc=function(t){q.DK(this,t);},y.a=0,y.t=0,y.v=false,l.N9=X(293),q.NK=function(){q.NK=o1,q.MK=F(k(l.Sbb,1),st,2,6,["None","One electron (duplet)","Two electrons (triplet)","Two electrons (singulet)"]);},q.OK=function(t){var e,A,P,r,i,D,u,o,Q;for(A=F(k(l.d7,1),H,6,15,[8,-2,8,-2,8]),Q=F(k(l.d7,1),H,6,15,[8,-2,4,-2,12,-2,4,-2,12,-2,4,-2,12,-2,8]),R.u2(t.c,A,Q),t.o=R.r2(t.c,1,1),R.Q1(t.o,t),R.m2(t.c,R.q2(t.c,"Atom Label:"),1,1),R.m2(t.c,t.o,3,1),R.n2(t.c,R.q2(t.c,"(examples: 'D', 'Li', 'Cys', 'R12', 'R3@C')"),1,3,3,3),t.p=R.r2(t.c,1,1),R.Q1(t.p,t),R.m2(t.c,R.q2(t.c,"Atom Mass:"),1,5),R.m2(t.c,t.p,3,5),R.n2(t.c,R.q2(t.c,"(empty for natural abundance)"),1,7,3,7),t.q=R.r2(t.c,1,1),R.Q1(t.q,t),R.m2(t.c,R.q2(t.c,"Abnormal Valence:"),1,9),R.m2(t.c,t.q,3,9),R.n2(t.c,R.q2(t.c,"(empty for default valence)"),1,11,3,11),t.a==-1?(P=(n.Qk(),n.Kk)[t.k],R.n3(t.o,t.n==null?P:t.n+"@"+P),t.g!=0&&R.n3(t.p,""+t.g),t.j!=-1&&R.n3(t.q,""+t.j)):(P=n.Hl(t.e,t.a),e=n.Dl(t.e,t.a),R.n3(t.o,e==null?P:e+"@"+P),n.Ll(t.e,t.a)!=0&&R.n3(t.p,""+n.Ll(t.e,t.a)),n.yl(t.e,t.a)!=-1&&R.n3(t.q,""+n.yl(t.e,t.a))),t.b=R.p2(t.c),i=q.MK,D=0,u=i.length;D<u;++D)r=i[D],R.e2(t.b,r);o=t.a==-1?t.i:n.Ol(t.e,t.a),R.j2(t.b,o==32?1:o==48?2:o==16?3:0),R.m2(t.c,R.q2(t.c,"Radical State:"),1,13),R.m2(t.c,t.b,3,13);},q.PK=function(t,e){e.j==1?(t.a!=-1&&(n.Bn(t.e,t.a,t.k),n.sn(t.e,t.a,t.g),n.fn(t.e,t.a,t.j),n.vn(t.e,t.a,t.i),n.ln(t.e,t.a,t.n)),R.s2(t.c)):e.j==0&&(q.QK(t),R.s2(t.c));},q.QK=function(t){var e,A,P,r,i,D,u;if(D=R.m3(t.o),A=null,w.Xzb(D).length!=0&&(P=T.ljb(D,Z1(64)),P!=-1&&(A=(w.Rzb(0,P,w.Xzb(D).length),w.Xzb(D).substr(0,P)),D=(w.Szb(P+1,w.Xzb(D).length+1),w.Xzb(D).substr(P+1)))),w.Xzb(D).length!=0&&(e=n.oo(D,t.d.c),e!=0||T.ejb(D,"?"))){if(r=0,w.Xzb(R.m3(t.p)).length!=0)try{if(r=T.Qhb(R.m3(t.p),P1,St),r<(n.Qk(),n.Ok)[e]-18||r>n.Ok[e]+12){R.w2(t.c,"Your mass is out of range!");return;}}catch(o){if(o=H1(o),Jt(o,40)){R.w2(t.c,"Your mass is not a number!");return;}else throw Ut(o);}if(u=-1,w.Xzb(R.m3(t.q)).length!=0)try{if(u=T.Qhb(R.m3(t.q),P1,St),u<0||u>15){R.w2(t.c,"Your valence is out of range!");return;}}catch(o){if(o=H1(o),Jt(o,40)){R.w2(t.c,"Your valence is not a number!");return;}else throw Ut(o);}i=R.f2(t.b)==1?32:R.f2(t.b)==2?48:R.f2(t.b)==3?16:0,q.$L(t.d,e,r,u,i,A),t.a!=-1&&(n.el(t.e,t.a,e,r,u,i),n.ln(t.e,t.a,A)),t.f=true,R.s2(t.c);}},q.RK=function(t){return t.f=false,R.v2(t.c),t.f;},q.SK=function(t,e,A,P,r,i,D){q.NK(),this.c=W.p1(t,FD,this),this.d=e,this.a=-1,this.k=A,this.g=P,this.j=r,this.i=i,this.n=D,q.OK(this);},q.TK=function(t,e,A,P){q.NK(),this.c=W.p1(t,FD,this),this.d=e,this.e=A,this.a=P,this.k=n.Sl(this.e,P),this.g=n.Ll(this.e,P),this.j=n.yl(this.e,P),this.i=n.Ol(this.e,P),this.n=n.Dl(this.e,P),q.OK(this);},x(220,1,hn,q.SK,q.TK),y.lc=function(t){q.PK(this,t);},y.a=0,y.f=false,y.g=0,y.i=0,y.j=0,y.k=0,l.O9=X(220),U.VK=function(t){this.k=t;},x(317,1,{}),l.Ocb=X(317),Tt.WK=function(t,e){U.VK.call(this,e),this.j=t;},x(161,317,{}),y.j=0,l.Y9=X(161),q.XK=function(t,e,A){Tt.WK.call(this,e,t),this.a=A;},x(24,161,{24:1},q.XK),y.a=false,l.P9=X(24),q.ZK=function(t,e){U.rj(t.S,e);},q.$K=function(t,e){var A,P,r,i;if(!e||e.q==0)return  false;if(e.r!=0)for(P=n.Ul(t.V,24*t.ab),n.ae(new Tt.YM(e),new R.W2(W.E0(t.k)),new Tt.FN(0,0,W.C0(t.k),W.B0(t.k)),q1+Ft(P));q.cL(t,e,.2*P);)n.$n(e,.5*P,.5*P);if(q.iM(t),n.Xm(e),n.$m(e),r=t.V.K,t.V.q==0)n.nl(e,t.V),t.a||n.Nn(t.V,r),t.db=j.Math.max(t.db,4),W.H0(t.k),q.zL(t,new q.XK(t,1,true));else {for(i=t.V.q,n.Xk(t.V,e),t.a||n.Nn(t.V,r),A=0;A<t.V.q;A++)n.wn(t.V,A,A>=i);t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));}return  true;},q._K=function(t,e){var A,P,r,i,D,u;for(D=(t.U&4)!=0,u=false,r=false,i=false,A=null,P=0;P<t.V.q;P++)if(n.Om(t.V,P))if(u){if(i!=tt.DH(A,n.Pl(t.V,P),n.Ql(t.V,P))){r=true;break;}}else {if(u=true,!D)break;A=U.wj(t.I,0),i=tt.DH(A,n.Pl(t.V,P),n.Ql(t.V,P));}return e?D?r?(q.nL(t,true),true):u?(q.mL(t,true),false):(q.nL(t,false),true):(q.mL(t,u),false):D&&(r||!u);},q.aL=function(t){var e,A;n.aw(t.V,15),e=z(l.d7,H,6,t.V.q,15,1),A=n.gp(t.V,e,false,true),A=q.ML(t,e,A),q.hM(t,e,A),t.M=e,t.L=n.hw(t.V,e,A);},q.bL=function(t,e,A){var P,r,i,D,u;if(t.V.K){if(n.Im(t.V,e)||n.Im(t.V,A)||(P=n.Il(t.V,e),r=n.Il(t.V,A),P==null^r==null))return  false;if(P!=null){if(P.length!=r.length)return  false;for(i=0;i<P.length;i++)if(P[i]!=r[i])return  false;}if(D=kt(G(n.Nl(t.V,e),1),0),u=kt(G(n.Nl(t.V,A),1),0),D!=u)return  false;}return n.Sl(t.V,e)==n.Sl(t.V,A);},q.cL=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h;for(P=0,A*=A,u=0;u<e.q;u++){for(Q=e.J[u].a,s=e.J[u].b,D=false,o=0;o<t.V.q;o++)if(L=n.Pl(t.V,o),h=n.Ql(t.V,o),r=L-Q,i=h-s,r*r+i*i<A){D=true;break;}D&&++P;}return P==e.q;},q.dL=function(t){var e,A,P;for(q.aL(t),a.QI(new a.SI(),q.EL(t)),P=z(l.d7,H,6,t.L.length,15,1),e=0;e<t.V.q;e++)A=t.M[e],n.qn(t.V,e,n.Kl(t.L[A],P[A]),n.Kl(t.V,e)==0||n.zm(t.V,e)),++P[A];},q.eL=function(t,e){switch(e){case 0:q.lL(t);return;case 1:q.iM(t),t.db=j.Math.max(t.db,6),W.H0(t.k),q.zL(t,new q.XK(t,1,true));return;case 17:q.XL(t),t.db=j.Math.max(t.db,2),W.H0(t.k),q.zL(t,new q.XK(t,1,true));return;}},q.fL=function(t,e){var A,P,r,i;for(P=0,r=0,i=0,A=0;A<t.V.q;A++)(!e||n.Om(t.V,A))&&(r+=n.Pl(t.V,A),i+=n.Ql(t.V,A),++P);return P>1?new z0.Zfb(r/P,i/P):null;},q.gL=function(t,e,A){var P,r,i,D,u;for(u=z(l.ebb,l1,158,A,0,1),D=z(l.d7,H,6,A,15,1),i=0;i<A;i++)u[i]=new z0.Yfb();for(P=0;P<t.V.q;P++)u[e[P]].a+=n.Pl(t.V,P),u[e[P]].b+=n.Ql(t.V,P),++D[e[P]];for(r=0;r<A;r++)u[r].a/=D[r],u[r].b/=D[r];return u;},q.hL=function(t,e){q.iM(t),n.Zk(t.V,t.gb,t.jb,e,0,-1,0,null)&&(t.t=-1,t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));},q.iL=function(t,e){var A,P,r;for(P=0,A=0;A<t.V.q;A++)n.Om(t.V,A)&&++P;r=P!=0&&P!=t.V.q,(t.U&1)!=0?q.kL(t,e,r):q.jL(t,e,r);},q.jL=function(t,e,A){var P;if(t.db==6){if(A)for(P=0;P<t.V.q;P++)n.rn(t.V,P,!n.Om(t.V,P));Z.iA(new Z.BA(A?4:0),t.V),A&&n.Ym(t.V);}n.zk(t.G,e,new Tt.FN(0,0,W.C0(t.k),W.B0(t.k)),q1|t.T);},q.kL=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;if(A&&t.db==6)for(d=z(l.d7,H,6,t.L.length,15,1),i=0;i<t.V.q;i++)h=t.M[i],n.rn(t.L[h],d[h],!n.Om(t.V,i)),++d[h];for(u=z(l.bab,C1,35,t.L.length,0,1),b=0;b<t.L.length;b++)t.db==6&&(Z.iA(new Z.BA(A?4:0),t.L[b]),n.xq(t.L[b])),o=new Tt.YM(t.L[b]),n.ae(o,e,null,q1|t.T),u[b]=o.v;for(C=1.8000000715255737*t.T,D=n.Ul(t.V,24*t.ab),P=(t.U&4)==0?0:t.db==5?LD*W.C0(t.k):tt.CH(U.wj(t.I,0))*t.T/D,m=.5*C,c=0;c<=t.L.length&&((t.U&4)!=0&&c==t.Y&&(tt.EH(U.wj(t.I,0),m-C/2,W.B0(t.k)/2,m-C/2+P,W.B0(t.k)/2),m+=P),c!=t.L.length);c++)Q=m-u[c].c,L=.5*(W.B0(t.k)-u[c].a)-u[c].d,n.$n(t.L[c],Q,L),m+=C+u[c].b;for(n.zk(t.G,e,new Tt.FN(0,0,W.C0(t.k),W.B0(t.k)),q1|t.T),g=z(l.d7,H,6,t.L.length,15,1),r=0;r<t.V.q;r++)s=t.M[r],n.yn(t.V,r,n.Pl(t.L[s],g[s])),n.zn(t.V,r,n.Ql(t.L[s],g[s])),++g[s];n.xq(t.V);},q.lL=function(t){var e;t.I&&(w.pzb(t.I.a,0),t.db=j.Math.max(t.db,1),W.H0(t.k)),q.iM(t),e=t.V.K,n.$v(t.V),n.Nn(t.V,e),t.cb.q!=0?(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true))):(t.db=j.Math.max(t.db,1),W.H0(t.k));},q.mL=function(t,e){return t.V.q!=0&&t.r?R.$1(t.r,e?q.FL(t.V):n.ew(t.V)):false;},q.nL=function(t,e){var A;return A=e?q.GL(t):q.EL(t),A&&t.r?R._1():false;},q.oL=function(t,e,A){var P;return n.ql(t.V,e,A)?(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true)),true):(P=q.xL(t,e,A),P?(U.Ej(t.I,P),t.v=null,t.db=j.Math.max(t.db,1),W.H0(t.k),true):false);},q.pL=function(t){return t.t!=-1?(n.pl(t.V,t.t),t.t=-1,t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true)),true):t.u!=-1?(n.tl(t.V,t.u),t.u=-1,t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true)),true):t.v&&!t.v.j?(U.Ej(t.I,t.v),t.v=null,t.db=j.Math.max(t.db,1),W.H0(t.k),true):false;},q.qL=function(t,e){var A,P,r;if(t.t!=-1&&(R.V2(e,(r=R.R2(e),r!=0?r:dP)),q.JL(t,e,t.t),t.w==19&&(P=n.Kl(t.V,t.t),P!=0)))for(A=0;A<t.V.f;A++)A!=t.t&&n.Kl(t.V,A)==P&&q.JL(t,e,A);t.u!=-1&&(R.V2(e,(r=R.R2(e),r!=0?r:dP)),q.KL(t,e,t.u)),t.v&&t.v.Lb(e);},q.rL=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c;for(i=0,A=0;A<t.V.q;A++)n.Om(t.V,A)&&++i;for(b=t.V.q,c=t.V.r,t.fb=U.Dqb(t.fb,t.fb.length+i),t.ib=U.Dqb(t.ib,t.ib.length+i),D=z(l.d7,H,6,t.V.q,15,1),o=n.bn(t.V,1),Q=n.bn(t.V,2),P=0;P<b;P++)n.Om(t.V,P)&&(s=t.V.q,t.fb[s]=t.fb[P],t.ib[s]=t.ib[P],D[P]=s,n.kl(t.V,t.V,P,o,Q));for(u=0;u<c;u++)n.Pm(t.V,u)&&n.ml(t.V,t.V,u,o,Q,D,false);for(r=0;r<b;r++)n.wn(t.V,r,false);for(e=b;e<t.V.q;e++)n.qn(t.V,e,0,false);if(t.I)for(L=t.I.a.length-1;L>=0;L--)h=U.wj(t.I,L),h.g&&!Jt(h,61)&&U.rj(t.I,h.Fb());},q.sL=function(t,e){var A,P;for(new q.PM(t._,e),P=false,A=0;A<w.Xzb(e.e).length;A++)if(!n2(T.Yib(e.e,A))){P=true;break;}P||U.Ej(t.I,e),W.H0(t.k);},q.tL=function(t){var e,A,P;if(null.xd()||null.xd())q._K(t,true);else if(null.xd())t.r&&R.b2();else if(null.xd())q.pM(t,null.xd());else if(null.xd())q.UL(t);else if(!null.xd())if(null.xd())q.dL(t),q.nM(t,j.Math.max(t.db,1));else if(null.xd())q.WL(t,true,true);else if(null.xd())q.WL(t,false,true);else if(null.xd())q.WL(t,true,false);else if(null.xd())q.PL(t);else if(null.xd())q.YK=!q.YK,q.YK&&q.dL(t);else if(null.xd())q.AL(t,true);else if(null.xd())q.AL(t,false);else if(null.xd())q.YL(t);else if(null.xd())W.z1(t._,vD,pD);else if(null.xd()){if(null.xd(),e=T.Qhb(null.xd(),P1,St),A=T.Qhb(null.xd(),P1,St),n.Om(t.V,e))for(P=0;P<t.V.f;P++)n.Om(t.V,P)&&n.jn(t.V,P,A);else n.jn(t.V,e,A);}else null.xd()?(t.H=6,t.db=j.Math.max(t.db,1),W.H0(t.k)):null.xd()?(t.H=256,t.db=j.Math.max(t.db,1),W.H0(t.k)):null.xd()&&q._L(t,t.w==19?16:0);},q.uL=function(t,e){var A,P,r,i,D,u,o,Q,L,s;if(!t.W){if(e.j==1)if(e.b==-3&&(t.Z=true,q.oM(t)),e.b==-2&&(t.d=true,q.oM(t)),e.b==-1&&q.oM(t),e.a)e.b==122?(q.XL(t),t.db=j.Math.max(t.db,2),W.H0(t.k),q.zL(t,new q.XK(t,1,true))):e.b==99?q._K(t,true):e.b==118&&((t.U&4)!=0&&t.r&&R.b2(),q.UL(t));else if(e.b==-4)t.w!=19&&(q.iM(t),q.pL(t)||n.vl(t.V)&&(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true))));else if(e.b==-6||t.t==-1&&e.b==63){W.z1(t._,vD,pD);return;}else if(e.b==-8)w.Xzb(t.i.a).length!=0&&(q.wL(t,t.i.a),T.Agb(t.i,0));else if(t.u!=-1)i=e.b,i==113&&t.V.K?q.fM(t,t.u):i==118?n.bl(t.V,t.u,3,false,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true))):i>=52&&i<=55?n.bl(t.V,t.u,i-48,false,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true))):i==97||i==98?n.bl(t.V,t.u,6,true,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true))):(r=i==48?(q.iM(t),n.hl(t.V,t.u,16)):i==49?(q.iM(t),n.hl(t.V,t.u,1)):i==50?(q.iM(t),n.hl(t.V,t.u,2)):i==51?(q.iM(t),n.hl(t.V,t.u,4)):i==117?(q.iM(t),n.hl(t.V,t.u,257)):i==100?(q.iM(t),n.hl(t.V,t.u,129)):i==99?(q.iM(t),n.hl(t.V,t.u,386)):i==109&&(q.iM(t),n.hl(t.V,t.u,16)),r&&(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true))));else if(t.t!=-1){if(i=e.b,Q=w.Xzb(t.i.a).length==0,Q?t.K=i:(t.K==108&&(T.Agb(t.i,0),T.Jjb(t.i,76)),t.K=-1),Q&&i==108)T.Njb(t.i,"Cl"),t.db=j.Math.max(t.db,1),W.H0(t.k);else if(Q&&(i==43||i==45))q.iM(t),n.gl(t.V,t.t,i==43)&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));else if(Q&&i==46)q.iM(t),s=n.Ol(t.V,t.t)==32?0:32,n.vn(t.V,t.t,s),t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));else if(Q&&i==58)q.iM(t),s=n.Ol(t.V,t.t)==48?16:n.Ol(t.V,t.t)==16?0:48,n.vn(t.V,t.t,s),t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));else if(Q&&i==113&&t.V.K)q.eM(t,t.t);else if(Q&&t.V.K&&(i==120||i==88))L=F(k(l.d7,1),H,6,15,[9,17,35,53]),n.on(t.V,t.t,L),t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));else if(Q&&i==63)q.iM(t),n.el(t.V,t.t,0,0,-1,0)&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));else if(Q&&i>48&&i<=57){if(n.ip(t.V,t.t)>0){for(q.iM(t),D=i-47,A=t.t,u=t.V.q-t.V.f,o=1;o<D&&(q.kM(t,A),P=n.Sk(t.V,t.hb,t.kb,0),P!=-1);o++)n.Uk(t.V,A,P),A=P-u,n.aw(t.V,1);t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));}}else !Q&&e.b==-7?(T.Agb(t.i,0),t.db=j.Math.max(t.db,1),W.H0(t.k)):!Q&&e.b==-5?(T.Agb(t.i,w.Xzb(t.i.a).length-1),t.db=j.Math.max(t.db,1),W.H0(t.k)):i>=65&&i<=90||i>=97&&i<=122||i>=48&&i<=57||i==45?(T.Jjb(t.i,i&r1),t.db=j.Math.max(t.db,1),W.H0(t.k)):(i==10||i==13)&&q.wL(t,t.i.a);}else (t.U&7)==0&&(i=e.b,i==104&&q.AL(t,true),i==118&&q.AL(t,false));e.j==2&&(e.b==-3&&(t.Z=false,q.oM(t)),e.b==-2&&(t.d=false,q.oM(t)),e.b==-1&&q.oM(t));}},q.vL=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C;if(e.j==1){if(t.t!=-1&&w.Xzb(t.i.a).length!=0&&q.wL(t,t.i.a),T.Agb(t.i,0),e.e){q.IL(t);return;}if(e.b==1){if(e.c==2)return;t.W=true,q.oM(t),q.RL(t,e);}}if(e.j==2){if(e.e){q.IL(t);return;}e.b==1&&(t.W=false,q.oM(t),q.SL(t));}if(e.j==3&&e.b==1&&e.c==2){q.HL(t,e.g,e.i);return;}if(e.j==4&&(W.v1(t._),q.oM(t)),e.j==6&&(t.W=false,m=e.g,C=e.i,q.mM(t,m,C,false)&&W.H0(t.k),q.oM(t)),e.j==7){switch(t.W=true,t.hb=e.g,t.kb=e.i,h=q.mM(t,t.hb,t.kb,true),t.X){case 2:q.jM(t)&&(h=true);break;case 1:(t.hb-t.gb)*(t.hb-t.gb)+(t.kb-t.jb)*(t.kb-t.jb)<100&&q.kM(t,t.e),h=true;break;case 3:n.yn(t.V,t.e,t.fb[t.e]+t.hb-t.gb),n.zn(t.V,t.e,t.ib[t.e]+t.kb-t.jb),t.f!=-1&&(n.yn(t.V,t.f,t.fb[t.f]+t.hb-t.gb),n.zn(t.V,t.f,t.ib[t.f]+t.kb-t.jb)),t.db=j.Math.max(t.db,2),W.H0(t.k);break;case 9:q.rL(t),t.X=4;case 4:if(t.I)for(i=new U.wqb(t.I);i.a<i.c.a.length;)r=U.vqb(i),r.g&&r.Nb(t.hb,t.kb);for(u=0;u<t.V.q;u++)n.Om(t.V,u)&&(n.yn(t.V,u,t.fb[u]+t.hb-t.gb),n.zn(t.V,u,t.ib[u]+t.kb-t.jb));t.db=j.Math.max(t.db,2),W.H0(t.k);break;case 11:U.rj(t.I,t.v.Fb()),t.X=10;case 10:t.v.Nb(t.hb,t.kb),t.db=j.Math.max(t.db,2),W.H0(t.k);break;case 7:for(b=false,P=0;P<t.V.q&&!b;P++)b=n.Om(t.V,P);if(c=false,t.I)for(o=0;o<t.I.a.length&&!c;o++)c=U.wj(t.I,o).g;if(L=j.Math.abs(t.kb-t.jb)<20?1:j.Math.exp((t.kb-t.jb)/100),A=j.Math.abs(t.hb-t.gb)<20?0:(t.hb-t.gb)/50,g=b||c,t.I&&(!g||c)){for(o=0;o<t.I.a.length;o++)(!g||U.wj(t.I,o).g)&&U.wj(t.I,o).Ob(L,A);t.db=j.Math.max(t.db,2),W.H0(t.k);}(!g||b)&&(n.ao(t.V,L,A,g),t.db=j.Math.max(t.db,2),W.H0(t.k));break;case 5:case 6:if(d=null,t.X==5){if(j.Math.abs(t.hb-Tt.uN(t.R,t.R.a-1))<3&&j.Math.abs(t.kb-Tt.vN(t.R,t.R.a-1))<3)break;Tt.wN(t.R),Tt.tN(t.R,t.hb,t.kb),Tt.tN(t.R,t.gb,t.jb),d=t.R;}else d=new Tt.FN(j.Math.min(t.gb,t.hb),j.Math.min(t.jb,t.kb),j.Math.abs(t.hb-t.gb),j.Math.abs(t.kb-t.jb));if(t.I)for(o=0;o<t.I.a.length;o++)s=U.wj(t.I,o),Q=n.ke(s,d),(!t.O||!t.Q[o])&&Q!=s.g&&(s.g=Q,t.db=j.Math.max(t.db,1));for(D=0;D<t.V.q;D++)Q=d.Gb(Ft(n.Pl(t.V,D)),Ft(n.Ql(t.V,D))),(!t.O||!t.P[D])&&Q!=n.Om(t.V,D)&&(n.wn(t.V,D,Q),t.db=j.Math.max(t.db,1));h=true;break;case 8:h=true;}h&&W.H0(t.k);}},q.wL=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b;if(T.Agb(t.i,0),P=n.oo(e,t.c),P!=0)if(q.iM(t),n.el(t.V,t.t,P,0,-1,0)){t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));return;}else t.db=j.Math.max(t.db,0),W.H0(t.k);if(t.j!=null&&w.Xzb(t.j).length!=0&&(e=t.j),h=(n.Qs(),L=(!n.Ps&&n.Rs(),U.jpb(n.Ps,T.qjb(w.Xzb(e).toLowerCase(),"-"))),L==null?null:n.Oq(new n.$q(false),L)),h){for(q.iM(t),o=new n.sw(),n.wo(o,t.V,t.t,null),s=n.Ul(o,t.ab*24),D=o.q,A=0;A<o.q;A++)o.u[A]|=A0;for(n.cl(o,h,0),Z.iA(new Z.BA(4),o),r=n.Pl(t.V,t.t)-s*o.J[0].a,i=n.Ql(t.V,t.t)-s*o.J[0].b,u=t.V.q,n.cl(t.V,h,t.t),b=t.V.q-u,Q=0;Q<b;Q++)n.yn(t.V,u+Q,s*o.J[D+Q].a+r),n.zn(t.V,u+Q,s*o.J[D+Q].b+i);n.xq(t.V),t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));}},q.xL=function(t,e,A){var P,r;if(t.I){for(r=new U.wqb(t.I);r.a<r.c.a.length;)if(P=U.vqb(r),P.Gb(e,A))return P;}return null;},q.yL=function(t){var e,A,P,r,i,D;for(i=-1,D=L0,e=0;e<t.V.q;e++)P=t.gb-n.Pl(t.V,e),r=t.jb-n.Ql(t.V,e),A=j.Math.sqrt(P*P+r*r),A<24&&D>A&&(D=A,i=t.M[e]);return i;},q.zL=function(t,e){var A;A=e.j,(A&t.J)==0&&(W.x1(t._,new q.sM(t,A,e)),t.J|=A);},q.AL=function(t,e){var A,P,r,i,D;for(D=false,P=0;P<t.V.q;P++)if(n.Om(t.V,P)){D=true;break;}if(i=q.fL(t,D),i){for(q.iM(t),A=0;A<t.V.q;A++)(!D||n.Om(t.V,A))&&(e?n.yn(t.V,A,2*i.a-n.Pl(t.V,A)):n.zn(t.V,A,2*i.b-n.Ql(t.V,A)));for(r=0;r<t.V.r;r++)(!D||n.Pm(t.V,r))&&(n.hm(t.V,r)==257?n.Ln(t.V,r,129):n.hm(t.V,r)==129&&n.Ln(t.V,r,257));t.db=j.Math.max(t.db,1),W.H0(t.k);}},q.BL=function(t,e){return n.oo(e,t.c)!=0?1:(t.j=n.Ss(e),t.j==null?q.LL(e)?3:4:w.Xzb(t.j).length==0?3:2);},q.CL=function(t,e){var A,P,r;if(A=n.Yl(t.V,0,e),n.Ml(t.V,A)!=0)return n.xm(t.V,A)||n.Ml(t.V,A)!=1&&n.Ml(t.V,A)!=2?-1:A;if(n.Po(t.V,A)==1){for(r=0;r<n.Zo(t.V,A);r++)if(n._o(t.V,A,r)==2&&(P=n.Yo(t.V,A,r),n.Po(t.V,P)==2&&(n.Ml(t.V,P)==1||n.Ml(t.V,P)==2)))return P;}return  -1;},q.DL=function(t,e){var A;return A=n.Go(t.V,n.Yl(t.V,0,e)),A!=-1&&n.fm(t.V,A)!=1&&n.fm(t.V,A)!=2&&(A=-1),A;},q.EL=function(t){var e,A;if((t.U&4)==0||t.L==null)return null;for(A=new tt.yH(),e=0;e<t.L.length;e++)e<t.Y?tt.cH(A,t.L[e]):tt.aH(A,t.L[e]);return A;},q.FL=function(t){var e,A,P,r,i,D,u;for(P=0,A=0;A<t.q;A++)(t.u[A]&512)!=0&&++P;if(P==0)return null;for(i=0,r=0;r<t.r;r++)(t.u[t.D[0][r]]&t.u[t.D[1][r]]&512)!=0&&++i;for(u=z(l.zeb,ht,6,t.q,16,1),e=0;e<t.q;e++)u[e]=(t.u[e]&512)!=0;return D=new n.tw(P,i),n.Bo(t,D,u,false,null),D;},q.GL=function(t){var e,A,P;for(A=new tt.yH(),e=0;e<t.L.length;e++)P=q.FL(t.L[e]),P&&(e<t.Y?(U.rj(A.g,P),A.d=-1):(U.rj(A.f,P),A.d=-1));return A;},q.HL=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m;if(i=n.wl(t.V,e,A),D=n.xl(t.V,e,A),t.w==2){if(t.V.K){if(i!=-1)q.eM(t,i);else if(D!=-1)q.fM(t,D);else if(t.v){if(!t.Z){for(h=0;h<t.V.q;h++)n.wn(t.V,h,false);for(r=new U.wqb(t.I);r.a<r.c.a.length;)P=U.vqb(r),P.g=false;}t.v.g=true,t.db=j.Math.max(t.db,1),W.H0(t.k);}}else if(m=-1,i!=-1?m=i:D!=-1&&(m=n.Yl(t.V,0,D)),m!=-1||t.v){if(!t.Z){for(h=0;h<t.V.q;h++)n.wn(t.V,h,false);if(t.I)for(Q=new U.wqb(t.I);Q.a<Q.c.a.length;)o=U.vqb(Q),o.g=false;}if(m!=-1){if((t.U&1)!=0)for(L=t.M[m],h=0;h<t.V.q;h++)t.M[h]==L&&n.wn(t.V,h,true);else for(s=n.ep(t.V,m,false),h=0;h<s.length;h++)n.wn(t.V,s[h],true);}else t.v.g=true;t.db=j.Math.max(t.db,1),W.H0(t.k);}}else if(t.w==18&&(L=-2,(t.U&1)!=0&&(L=q.yL(t)),L!=-1)){for(d=St,g=P1,b=0;b<t.V.q;b++)(L==-2||t.M[b]==L)&&(d>n.Pl(t.V,b)&&(d=n.Pl(t.V,b)),g<n.Pl(t.V,b)&&(g=n.Pl(t.V,b)));if(g>d){for(u=(g+d)/2,c=0;c<t.V.q;c++)(L==-2||t.M[c]==L)&&n.yn(t.V,c,2*u-n.Pl(t.V,c));for(h=0;h<t.V.r;h++)if(L==-2||t.M[n.Yl(t.V,0,h)]==L)switch(n.hm(t.V,h)){case 257:n.Ln(t.V,h,129);break;case 129:n.Ln(t.V,h,257);}}t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true));}},q.IL=function(t){t.r&&(q._K(t,false),null.xd(),null.xd(),(t.U&4)!=0&&(null.xd(),null.xd()),(t.U&4)!=0&&null.xd()),(t.U&4)!=0&&t.w==19&&(null.xd(),null.xd(),null.xd(),null.xd(),null.xd(),null.xd()),t.w==18&&(null.xd(),null.xd(),null.xd()),t.g&&t.t!=-1&&(n.Bl(t.V,t.t),null.xd(),null.xd(),null.xd(),null.xd(),null.xd(),null.xd(),null.xd(),null.xd(),null.xd(),null.xd()),null.xd();},q.JL=function(t,e,A){var P,r,i;P=Ft(pr*n.Tl(t.V)),r=Ft(n.Pl(t.V,A)),i=Ft(n.Ql(t.V,A)),R.J2(e,r-P,i-P,2*P);},q.KL=function(t,e,A){var P,r,i,D,u,o;r=Ft(pr*n.Tl(t.V)),i=Ft(n.Pl(t.V,n.Yl(t.V,0,A))),u=Ft(n.Ql(t.V,n.Yl(t.V,0,A))),D=Ft(n.Pl(t.V,n.Yl(t.V,1,A))),o=Ft(n.Ql(t.V,n.Yl(t.V,1,A))),P=R.Q2(e),R.U2(e,r),R.F2(e,i,u,D,o),R.U2(e,P);},q.LL=function(t){var e;if(w.Xzb(t).length<3){for(e=1;e<(n.Qk(),n.Kk).length;e++)if(T.tjb(n.Kk[e],t))return  true;}return  false;},q.ML=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O;if(A<2)return A;for(S=z(l.zeb,l1,12,A,0,2),c=1;c<A;c++)S[c]=z(l.zeb,ht,6,c,16,1);for(D=n.Ul(t.V,24*t.ab),r=1;r<t.V.q;r++)for(i=0;i<r;i++)o=n.Pl(t.V,i)-n.Pl(t.V,r),Q=n.Ql(t.V,i)-n.Ql(t.V,r),u=j.Math.sqrt(o*o+Q*Q),u<bA*D&&(s=e[r],h=e[i],s!=h&&(s>h?S[s][h]=true:S[h][s]=true));for(O=z(l.d7,H,6,A,15,1),L=0;L<A;L++)O[L]=L;for(M=0,b=1;b<A;b++)for(m=0;m<b;m++)if(S[b][m]&&(g=O[b],d=O[m],g!=d))for(++M,V=j.Math.min(g,d),v=j.Math.max(g,d),C=0;C<A;C++)O[C]==v?O[C]=V:O[C]>v&&--O[C];for(P=0;P<t.V.q;P++)e[P]=O[e[P]];return A-M;},q.NL=function(t,e,A){var P,r;for(t.J&=~e,r=new U.wqb(t.S);r.a<r.c.a.length;)P=U.vqb(r),P.lc(A);},q.OL=function(t,e,A,P){return (t.U&6)!=0&&A[1]!=P[1]?A[1]==0?-1:1:Ge(e[A[0]].a+e[A[0]].b,e[P[0]].a+e[P[0]].b);},q.PL=function(t){var e,A;for(A=false,e=0;e<t.V.f;e++)n.Kl(t.V,e)!=0&&n.zm(t.V,e)&&(A||(q.iM(t),A=true),n.qn(t.V,e,n.Kl(t.V,e),false));A&&q.nM(t,j.Math.max(t.db,1));},q.QL=function(t){t.db=j.Math.max(t.db,4),W.H0(t.k),q.zL(t,new q.XK(t,1,false));},q.RL=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c;switch(t.gb=e.g,t.jb=e.i,t.w){case 18:if(b=t.gb,c=t.jb,t.e=n.wl(t.V,t.gb,t.jb),t.e!=-1&&(t.gb=n.Pl(t.V,t.e),t.jb=n.Ql(t.V,t.e)),n.bo(t.V,b,c),t.I)for(u=new U.wqb(t.I);u.a<u.c.a.length;)D=U.vqb(u),D.Pb(b,c);q.iM(t),t.X=7;break;case 2:if(t.X=0,t.e=n.wl(t.V,t.gb,t.jb),t.e!=-1&&(t.f=-1,t.gb=n.Pl(t.V,t.e),t.jb=n.Ql(t.V,t.e),n.Om(t.V,t.e)?t.X=t.Z?9:4:t.X=3),t.X==0&&(r=n.xl(t.V,t.gb,t.jb),r!=-1&&(t.e=n.Yl(t.V,0,r),t.f=n.Yl(t.V,1,r),n.Pm(t.V,r)?t.X=t.Z?9:4:t.X=3)),t.X==0&&t.v&&(t.v.g?t.X=t.Z?9:4:t.X=t.Z&&!Jt(t.v,61)?11:10),t.X!=0){for(t.fb=z(l.b7,jt,6,t.V.q,15,1),t.ib=z(l.b7,jt,6,t.V.q,15,1),Q=0;Q<t.V.q;Q++)t.fb[Q]=n.Pl(t.V,Q),t.ib[Q]=n.Ql(t.V,Q);if(t.I)for(u=new U.wqb(t.I);u.a<u.c.a.length;)D=U.vqb(u),n.oe(D,t.gb,t.jb);q.iM(t);break;}for(t.P=z(l.zeb,ht,6,t.V.q,16,1),t.I&&(t.Q=z(l.zeb,ht,6,t.I.a.length,16,1)),t.O=e.f,L=0;L<t.V.q;L++)t.P[L]=n.Om(t.V,L);if(t.I)for(o=0;o<t.I.a.length;o++)t.Q[o]=U.wj(t.I,o).g;e.a?t.X=6:(t.R=new Tt.xN(),Tt.tN(t.R,t.gb,t.jb),Tt.tN(t.R,t.gb,t.jb),t.X=5);break;case 4:q.iM(t),q.oL(t,t.gb,t.jb);break;case 3:h=n.wl(t.V,t.gb,t.jb),h!=-1&&(q.iM(t),n.kn(t.V,h,!n.vm(t.V,h)),t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 101:case 103:case 102:t.u!=-1&&q.VL(t,t.u)&&(q.iM(t),q.aM(t,t.u,t.w==101?0:t.w==103?1:2),t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 5:case 6:case 23:if(t.e=n.wl(t.V,t.gb,t.jb),t.e==-1){if(P=n.xl(t.V,t.gb,t.jb),P!=-1){q.iM(t),i=511,t.w==6?i=257:t.w==23&&(i=129),n.hl(t.V,P,i)&&(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;}}else {if(n.Lo(t.V,t.e)==8)return;t.gb=n.Pl(t.V,t.e),t.jb=n.Ql(t.V,t.e);}t.X=1,q.kM(t,t.e),W.H0(t.k);break;case 22:if(t.e=n.wl(t.V,t.gb,t.jb),t.e!=-1){if(n.Lo(t.V,t.e)==8)return;t.gb=n.Pl(t.V,t.e),t.jb=n.Ql(t.V,t.e);}t.X=2,t.q=0;break;case 7:q.iM(t),n._k(t.V,t.gb,t.jb,3,false,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 24:q.iM(t),n._k(t.V,t.gb,t.jb,4,false,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 8:q.iM(t),n._k(t.V,t.gb,t.jb,5,false,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 25:q.iM(t),n._k(t.V,t.gb,t.jb,6,false,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 9:q.iM(t),n._k(t.V,t.gb,t.jb,7,false,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 26:q.iM(t),n._k(t.V,t.gb,t.jb,6,true,n.Ul(t.V,24*t.ab))&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 10:q.iM(t),n.fl(t.V,t.gb,t.jb,true)&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 27:q.iM(t),n.fl(t.V,t.gb,t.jb,false)&&(t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));break;case 16:q.hL(t,1);break;case 11:q.hL(t,6);break;case 12:q.hL(t,7);break;case 13:q.hL(t,8);break;case 28:q.hL(t,14);break;case 29:q.hL(t,15);break;case 30:q.hL(t,16);break;case 14:q.hL(t,9);break;case 31:q.hL(t,17);break;case 15:q.hL(t,35);break;case 32:q.hL(t,53);break;case 33:e.d||e.a?(A=n.wl(t.V,t.gb,t.jb),A!=-1&&q.gM(t,A)):(q.iM(t),n.Zk(t.V,t.gb,t.jb,t.F,t.B,t.D,t.C,t.A)&&(t.t=-1,t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true))));break;case 19:t.e=n.wl(t.V,t.gb,t.jb),t.e!=-1&&t.e<t.V.f&&(!t.V.K||!n.Im(t.V,t.e))&&(t.gb=n.Pl(t.V,t.e),t.jb=n.Ql(t.V,t.e),t.X=8);break;case 21:s=null,t.v?Jt(t.v,94)&&(s=t.v):(s=new n._w(),n.Vw(s,t.gb,t.jb),U.rj(t.I,s)),q.sL(t,s),q.iM(t),t.db=j.Math.max(t.db,3),W.H0(t.k);}},q.SL=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g;switch(h=t.X,t.X=0,h){case 1:if(c=n.wl(t.V,t.hb,t.kb),c!=-1&&n.Lo(t.V,c)==8)return;q.iM(t),t.e==-1&&(t.e=n.Sk(t.V,t.gb,t.jb,0)),c==-1&&(c=n.Sk(t.V,t.hb,t.kb,0)),t.e!=-1&&c!=-1&&(r=n.Km(t.V,t.e)||n.Km(t.V,c)?16:1,t.w==6&&(r=257),t.w==23&&(r=129),n.$k(t.V,t.e,c,r)),t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));break;case 2:if(q.iM(t),t.q>0&&(t.e==-1&&(t.e=n.Sk(t.V,t.gb,t.jb,0)),t.n[0]==-1&&(t.n[0]=n.Sk(t.V,t.o[0],t.p[0],0)),t.n[0]!=-1&&n.Uk(t.V,t.e,t.n[0])),t.q>1)for(u=1;u<t.q;u++)t.n[u]==-1&&(t.n[u]=n.Sk(t.V,t.o[u],t.p[u],0)),t.n[u]!=-1&&n.Uk(t.V,t.n[u-1],t.n[u]);t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));break;case 3:case 4:case 7:t.db=j.Math.max(t.db,3),W.H0(t.k),q.zL(t,new q.XK(t,1,true));break;case 10:t.db=j.Math.max(t.db,3),W.H0(t.k);break;case 5:case 6:for(b=false,D=0;D<t.V.q;D++)if(t.P[D]!=n.Om(t.V,D)){b=true;break;}W.H0(t.k),b&&q.zL(t,new q.XK(t,2,true));break;case 8:if(s=false,P=t.t,P!=-1&&t.V.K&&n.Im(t.V,P)&&(P=-1),Q=n.Kl(t.V,t.e),P==-1){if(q.iM(t),Q!=0){for(s=true,g=false,e=0;e<t.V.f;e++)n.Kl(t.V,e)==Q&&(g=g|n.zm(t.V,e),n.qn(t.V,e,0,false));q.YK&&!g&&q.dL(t);}}else {if(q.iM(t),s=true,g=false,t.e==P)for(o=n.Kl(t.V,t.e),e=0;e<t.V.f;e++)n.Kl(t.V,e)==o&&(g=g|n.zm(t.V,e),n.qn(t.V,e,0,false));else {for(L=n.Kl(t.V,P),A=0;A<t.V.f;A++)(n.Kl(t.V,A)==Q||n.Kl(t.V,A)==L)&&n.qn(t.V,A,0,false);for(i=1,e=0;e<t.V.f;e++)n.Kl(t.V,e)==i&&(++i,e=-1);n.qn(t.V,t.e,i,false),n.qn(t.V,P,i,false);}q.YK&&!g&&q.dL(t);}s&&q.nM(t,j.Math.max(t.db,1)),W.H0(t.k);}},q.TL=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M,S,V,O,K,N,I,ut,Qt,Dt,Lt,ot,rt,yt,bt,it,ct;if((t.eb!=W.C0(t.k)||t.N!=W.B0(t.k))&&(t.eb=W.C0(t.k),t.N=W.B0(t.k),t.db<3&&(t.db=3)),r=R.M2(e),u=R.P2(e),R.V2(e,r),R.L2(e,0,0,t.eb,t.N),(t.U&4)!=0&&t.I.a.length==0&&(b=.5*t.eb,c=.5*t.N,i=.03999999910593033*t.eb,A=new tt.FH(),tt.EH(A,b-i,c,b+i,c),A.j=true,U.rj(t.I,A)),L=false,t.db!=0){switch((t.U&1)!=0&&t.db!=5&&q.aL(t),t.G=(t.U&4)!=0?new n.Ck(new tt.AH(t.L,t.Y),t.I):(t.U&2)!=0?new n.Dk(t.L,t.Y,t.I):(t.U&1)!=0?new n.Ek(t.L,t.I):new n.Bk(t.V,t.I),n.xk(t.G,u,r),n.uk(t.G,t.T),n.yk(t.G,(t.U&1)==0?0:(s=jP,T.ejb(w.Xzb(s).substr(0,18),Nn)||T.ejb(w.Xzb(s).substr(0,17),In)||T.ejb(w.Xzb(s).substr(0,14),xn)?lt.J3(r,TA):(lt.I3(),r&O0|Ft(j.Math.round(TA*((r&W0)>>16)))<<16|Ft(j.Math.round(TA*((r&J0)>>8)))<<8|Ft(j.Math.round(TA*(r&255)))))),n.wk(t.G,t.$),n.vk(t.G,t.H|8|(t.w==19?80:0)),t.db){case 6:case 4:case 5:q.iL(t,e);break;case 3:S=n.zk(t.G,e,new Tt.FN(0,0,t.eb,t.N),0),S&&(t.U&1)!=0&&n.ui(S,t.V);break;case 2:V=n.Ak(t.G,e,new Tt.FN(0,0,t.eb,t.N),0),L=!!V&&!(V.c==1&&V.a==0&&V.b==0);}t.db=0;}switch(t.G&&n.sk(t.G,e),L||q.qL(t,e),t.G&&(n.tk(t.G,e),n.rk(t.G,e)),t.t!=-1&&w.Xzb(t.i.a).length!=0&&(N=Ft(n.Pl(t.V,t.t)),ot=Ft(n.Ql(t.V,t.t)),d=t.i.a,O=q.BL(t,d),R.V2(e,O==1?u:O==2?-16776961:EA),O==1?d=(n.Qk(),n.Kk)[n.oo(d,t.c)]:O==2&&(d=T.vjb(t.j,0,w.Xzb(d).length)),D=3*R.O2(e)/2|0,R.T2(e,D,false,false),R.I2(e,N,ot,d),O==4&&(R.V2(e,fe),R.C2(e,N+R.N2(e,d).b/2,ot+D,"<unknown>")),O==2&&(R.V2(e,EA),R.I2(e,N+R.N2(e,d).b,ot,T.ujb(t.j,w.Xzb(d).length)))),R.V2(e,u),t.X){case 1:switch(I=Ft(t.gb),rt=Ft(t.jb),t.t==-1||t.t==t.e?(ut=Ft(t.hb),yt=Ft(t.kb)):(ut=Ft(n.Pl(t.V,t.t)),yt=Ft(n.Ql(t.V,t.t))),t.w){case 5:R.F2(e,I,rt,ut,yt);break;case 6:Qt=(rt-yt)/9|0,bt=(ut-I)/9|0,g=new Tt.yN(3),Tt.tN(g,I,rt),Tt.tN(g,ut+Qt,yt+bt),Tt.tN(g,ut-Qt,yt-bt),R.K2(e,new R.i3(g));break;case 23:for(Qt=ut-I,bt=yt-rt,Q=2;Q<17;Q+=2)Dt=I+(Q*Qt/17|0)-(Q*bt/128|0),it=rt+(Q*bt/17|0)+(Q*Qt/128|0),Lt=I+(Q*Qt/17|0)+(Q*bt/128|0),ct=rt+(Q*bt/17|0)-(Q*Qt/128|0),R.F2(e,Dt,it,Lt,ct);}break;case 2:if(t.q>0&&R.F2(e,Ft(t.gb),Ft(t.jb),Ft(t.o[0]),Ft(t.p[0])),t.q>1)for(Q=1;Q<t.q;Q++)R.F2(e,Ft(t.o[Q-1]),Ft(t.p[Q-1]),Ft(t.o[Q]),Ft(t.p[Q]));break;case 5:R.V2(e,(C=(v=R.R2(e),v!=0?v:dP),lt.L3(C,(h=jP,T.ejb(w.Xzb(h).substr(0,w.Xzb(Nn).length),Nn)||T.ejb(w.Xzb(h).substr(0,w.Xzb(In).length),In)||T.ejb(w.Xzb(h).substr(0,w.Xzb(xn).length),xn)?RD:CD)))),R.G2(e,new R.i3(t.R)),R.V2(e,u);break;case 6:N=t.gb<t.hb?Ft(t.gb):Ft(t.hb),ot=t.jb<t.kb?Ft(t.jb):Ft(t.kb),K=Ft(j.Math.abs(t.hb-t.gb)),o=Ft(j.Math.abs(t.kb-t.jb)),R.V2(e,(M=(m=R.R2(e),m!=0?m:dP),lt.L3(M,(s=jP,T.ejb(w.Xzb(s).substr(0,w.Xzb(Nn).length),Nn)||T.ejb(w.Xzb(s).substr(0,w.Xzb(In).length),In)||T.ejb(w.Xzb(s).substr(0,w.Xzb(xn).length),xn)?RD:CD)))),R.H2(e,N,ot,K,o),R.V2(e,u);break;case 8:I=Ft(t.gb),rt=Ft(t.jb),t.t==-1||t.t==t.e?(ut=Ft(t.hb),yt=Ft(t.kb)):(ut=Ft(n.Pl(t.V,t.t)),yt=Ft(n.Ql(t.V,t.t))),R.V2(e,(P=R.M2(e),lt.N3(-8388608,P))),R.F2(e,I,rt,ut,yt),R.V2(e,u);}},q.UL=function(t){var e;return !!(t.r&&(e=R.a2(t.r),q.$K(t,e)));},q.VL=function(t,e){return n.Qm(t.V,e)&&(q.CL(t,e)!=-1||q.DL(t,e)!=-1);},q.WL=function(t,e,A){var P,r;for(r=false,P=0;P<t.V.f;P++)n.Kl(t.V,P)!=0&&(A&&!n.zm(t.V,P)||e&&n.zm(t.V,P))&&(r||(q.iM(t),r=true),n.qn(t.V,P,0,false));r&&(e||q.dL(t),q.nM(t,j.Math.max(t.db,1)));},q.XL=function(t){t.cb&&(n.nl(t.cb,t.V),t.I=t.bb?new n.fk(t.bb):null);},q.YL=function(t){var e,A,P,r;for(r=false,A=0;A<t.V.q;A++)if(n.Om(t.V,A)){r=true;break;}if(P=q.fL(t,r),P){for(q.iM(t),e=0;e<t.V.q;e++)(!r||n.Om(t.V,e))&&(n.yn(t.V,e,2*P.a-n.Pl(t.V,e)),n.zn(t.V,e,2*P.b-n.Ql(t.V,e)));t.db=j.Math.max(t.db,1),W.H0(t.k);}},q.ZL=function(t,e){t.r=e;},q.$L=function(t,e,A,P,r,i){t.F=e,t.B=A,t.D=P,t.C=r,t.A=i;},q._L=function(t,e){t.H=e,t.db=j.Math.max(t.db,1),W.H0(t.k);},q.aM=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b,c;if(u=-1,r=q.CL(t,e),D=r==-1?q.DL(t,e):-1,A!=0){for(b=-1,L=0;L<t.V.f;L++)L!=r&&n.Gl(t.V,L)==A&&(!n.Pm(t.V,e)||!n.Om(t.V,L))&&(o=n.Fl(t.V,L),b<o&&(b=o));for(s=0;s<t.V.g;s++)s!=D&&n.bm(t.V,s)==A&&(!n.Pm(t.V,e)||!n.Pm(t.V,s))&&(o=n.am(t.V,s),b<o&&(b=o));if((r==-1?n.bm(t.V,D):n.Gl(t.V,r))!=A)u=j.Math.min(b+1,31);else if(u=r==-1?n.am(t.V,D):n.Fl(t.V,r),n.Pm(t.V,e)){for(c=true,h=0;h<t.V.f;h++)if(h!=r&&n.Om(t.V,h)&&n.Gl(t.V,h)==A&&n.Fl(t.V,h)!=u){c=false;break;}for(Q=0;Q<t.V.g;Q++)if(Q!=D&&n.Pm(t.V,Q)&&n.bm(t.V,Q)==A&&n.am(t.V,Q)!=u){c=false;break;}c&&(u<=b?(++u,u==32&&(u=0)):u=0);}else u<=b?(++u,u==32&&(u=0)):u=0;}if(n.Pm(t.V,e))for(Q=0;Q<t.V.g;Q++)n.Pm(t.V,Q)&&n.Qm(t.V,Q)&&(P=q.CL(t,Q),i=q.DL(t,Q),P!=-1?n.nn(t.V,P,A,u):i!=-1&&n.Fn(t.V,i,A,u));else r!=-1?n.nn(t.V,r,A,u):D!=-1&&n.Fn(t.V,D,A,u);},q.bM=function(t,e){t.V!=e&&(q.iM(t),t.V=e,t.U=0,t.I=null,t.db=j.Math.max(t.db,4),W.H0(t.k),q.zL(t,new q.XK(t,1,false)));},q.cM=function(t,e){var A,P,r,i;for(n.$v(t.V),t.L=z(l.d8,xe,25,e.g.a.length+e.f.a.length,0,1),t.Y=e.g.a.length,r=0;r<e.g.a.length+e.f.a.length;r++)t.L[r]=r<e.g.a.length?U.wj(e.g,r):U.wj(e.f,r-e.g.a.length),n.Xk(t.V,t.L[r]);for(n.Nn(t.V,e.c||tt.fH(e)),q.iM(t),t.M=z(l.d7,H,6,t.V.q,15,1),A=0,P=0;P<t.L.length;P++)for(i=0;i<t.L[P].q;i++)t.M[A++]=P;t.I=new n.ek(),t.U=5,t.db=j.Math.max(t.db,5),W.H0(t.k),q.zL(t,new q.XK(t,1,false));},q.dM=function(t,e,A){var P;return P=U.wj(t.I,0),tt.DH(P,n.Pl(t.V,e),n.Ql(t.V,e))==tt.DH(P,n.Pl(t.V,A),n.Ql(t.V,A));},q.eM=function(t,e){var A,P;t.b&&(q.iM(t),P=(t.U&4)!=0,A=new q.zK(t._,t.V,e,P),q.yK(A)&&(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true))));},q.fM=function(t,e){var A;t.b&&(q.iM(t),A=new q.KK(t._,t.V,e),q.JK(A)&&(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true))));},q.gM=function(t,e){var A;q.iM(t),A=e==-1?new q.SK(t._,t,t.F,t.B,t.D,t.C,t.A):new q.TK(t._,t,t.V,e),q.RK(A)&&e!=-1&&(t.db=j.Math.max(t.db,1),W.H0(t.k),q.zL(t,new q.XK(t,1,true)));},q.hM=function(t,e,A){var P,r,i,D,u,o,Q,L,s,h,b;for(s=h1(l.d7,[S1,H],[7,6],15,[A,(t.U&6)!=0?2:1],2),o=0;o<A;o++)s[o][0]=o;if(L=q.gL(t,e,A),(t.U&4)!=0)for(t.Y=0,P=(t.U&4)!=0?U.wj(t.I,0):null,Q=0;Q<A;Q++)s[Q][1]=tt.DH(P,L[Q].a,L[Q].b)?1:0,s[Q][1]==0&&++t.Y;else if((t.U&2)!=0)for(t.Y=A,r=0;r<t.V.q;r++)n.Sl(t.V,r)==0&&s[e[r]][1]==0&&(s[e[r]][1]=1,--t.Y);for(D=L,U.$qb(s,new q.vM(t,D)),h=z(l.d7,H,6,A,15,1),u=0;u<A;u++)b=s[u][0],h[b]=u;for(i=0;i<t.V.q;i++)e[i]=h[e[i]];},q.iM=function(t){!t.cb&&(t.cb=new n.co()),n.nl(t.V,t.cb),t.bb=t.I?new n.fk(t.I):null;},q.jM=function(t){var e,A,P,r,i,D,u,o,Q,L,s,h,b,c,g,d,m,C,v,M;if(v=n.mo(t.gb,t.jb,t.hb,t.kb),m=t.hb-t.gb,C=t.kb-t.jb,s=t.q,h=0,c=0,b=0,g=0,s>0&&(h=Gt(w1(j.Math.round(t.o[0]))),c=Gt(w1(j.Math.round(t.p[0])))),s>1&&(b=Gt(w1(j.Math.round(t.o[1]))),g=Gt(w1(j.Math.round(t.p[1])))),o=0,t.e==-1||n.Lo(t.V,t.e)==0)o=b0*nn(w1(j.Math.round(v*3/n1)));else if(n.Lo(t.V,t.e)==1)A=n.Xl(t.V,n.Yo(t.V,t.e,0),t.e),P=A-b0,r=A+b0,o=j.Math.abs(n.no(v,P))<j.Math.abs(n.no(v,r))?P:r;else {for(i=z(l.b7,jt,6,n.Lo(t.V,t.e),15,1),L=0;L<n.Lo(t.V,t.e);L++)i[L]=n.Xl(t.V,t.e,n.Yo(t.V,t.e,L));for(w.qzb(i,w.Xzb(h0(U.arb.prototype.jd,U.arb,[]))),Q=0;Q<i.length;Q++){if(d=Q==0?i[i.length-1]-X1:i[Q-1],d<v&&v<i[Q]){o=(i[Q]+d)/2;break;}if(d<v-X1&&v-X1<i[Q]){o=(i[Q]+d)/2;break;}}}if(e=n.Ul(t.V,24*t.ab),t.q=j.Math.abs(n.no(v,o))>b0?0:Ft(j.Math.sqrt(m*m+C*C)/e),t.q>0){for((t.o==null||t.o.length<t.q)&&(t.o=z(l.b7,jt,6,t.q,15,1),t.p=z(l.b7,jt,6,t.q,15,1)),D=z(l.b7,jt,6,2,15,1),u=z(l.b7,jt,6,2,15,1),M=n.no(v,o)<0?o-b0:o+b0,D[0]=e*j.Math.sin(o),u[0]=e*j.Math.cos(o),D[1]=e*j.Math.sin(M),u[1]=e*j.Math.cos(M),L=0;L<t.q;L++)t.o[L]=(L==0?t.gb:t.o[L-1])+D[L&1],t.p[L]=(L==0?t.jb:t.p[L-1])+u[L&1];for(t.n=z(l.d7,H,6,t.q,15,1),Q=0;Q<t.q;Q++)t.n[Q]=n.wl(t.V,t.o[Q],t.p[Q]),t.n[Q]!=-1&&(t.o[Q]=n.Pl(t.V,t.n[Q]),t.p[Q]=n.Ql(t.V,t.n[Q]));}return s!=t.q||t.q!=0&&(h!=Gt(w1(j.Math.round(t.o[0])))||c!=Gt(w1(j.Math.round(t.p[0]))))||t.q>1&&(b!=Gt(w1(j.Math.round(t.o[1])))||g!=Gt(w1(j.Math.round(t.p[1]))));},q.kM=function(t,e){var A,P,r,i,D,u,o,Q,L,s,h;if(s=En,e!=-1){for(A=z(l.b7,jt,6,9,15,1),D=0;D<n.Lo(t.V,e);D++)A[D]=n.Xl(t.V,e,n.Yo(t.V,e,D));if(n.Lo(t.V,e)==1)A[0]<-2.6179938779914944?s=b0:A[0]<uA?s=En:A[0]<qr?s=b0:A[0]<0?s=En:A[0]<Hn?s=rr:A[0]<m0?s=cA:A[0]<2.6179938779914944?s=rr:s=cA;else {for(u=n.Lo(t.V,e)-1;u>0;u--)for(o=0;o<u;o++)A[o]>A[o+1]&&(h=A[o],A[o]=A[o+1],A[o+1]=h);for(A[n.Lo(t.V,e)]=A[0]+X1,L=0,Q=0,i=0;i<n.Lo(t.V,e);i++)P=A[i+1]-A[i],Q<P&&(Q=P,L=i);s=(A[L]+A[L+1])/2;}}r=n.Ul(t.V,24*t.ab),t.hb=(e==-1?t.gb:n.Pl(t.V,e))+r*j.Math.sin(s),t.kb=(e==-1?t.jb:n.Ql(t.V,e))+r*j.Math.cos(s);},q.lM=function(t,e){t.w!=e&&(t.w==19||e==19?(t.w=e,t.db=j.Math.max(t.db,1),W.H0(t.k)):t.w=e);},q.mM=function(t,e,A,P){var r,i,D,u,o,Q;if(D=n.wl(t.V,e,A),u=-1,P&&t.X==8&&D!=-1&&(!q.bL(t,t.e,D)||n.Kl(t.V,t.e)!=0&&n.Kl(t.V,t.e)==n.Kl(t.V,D)&&!n.zm(t.V,t.e)||q.dM(t,t.e,D))&&(D=-1),D!=-1&&(t.w==101||t.w==103||t.w==102?(u=n.Hp(t.V,D),D=-1):t.w==19&&D>=t.V.f&&(D=-1)),u==-1&&D==-1&&t.w!=22&&t.w!=19&&t.w!=3&&t.w!=10&&t.w!=27&&t.w!=16&&t.w!=11&&t.w!=12&&t.w!=13&&t.w!=28&&t.w!=29&&t.w!=30&&t.w!=14&&t.w!=31&&t.w!=15&&t.w!=32&&t.w!=33&&(u=n.xl(t.V,e,A)),u!=-1&&(t.w==101||t.w==103||t.w==102)&&!(n.Qm(t.V,u)&&(q.CL(t,u)!=-1||q.DL(t,u)!=-1))&&(u=-1),r=t.v,!P&&t.I&&(r=null,D==-1&&u==-1&&(t.w==2||t.w==4||t.w==21))){for(Q=new U.wqb(t.I);Q.a<Q.c.a.length;)if(o=U.vqb(Q),(t.w==2||t.w==4&&!Jt(o,61)||t.w==21&&Jt(o,94))&&o.Db(e,A)){r=o,t.v&&t.v!=o&&t.v.Eb();break;}}return i=t.t!=D||t.u!=u||t.v!=r||!!r,t.t!=D&&(t.t!=-1&&w.Xzb(t.i.a).length!=0&&q.wL(t,t.i.a),t.t=D,T.Agb(t.i,0),q.zL(t,new q.XK(t,4,true))),t.u!=u&&(t.u=u,q.zL(t,new q.XK(t,8,true))),t.v=r,i;},q.nM=function(t,e){t.db=j.Math.max(t.db,e),W.H0(t.k),q.zL(t,new q.XK(t,1,true));},q.oM=function(t){var e;switch(e=-1,t.w){case 18:e=9;break;case 2:t.t!=-1&&n.Om(t.V,t.t)||t.u!=-1&&n.Pm(t.V,t.u)?e=t.W?4:t.Z?3:2:t.t!=-1||t.u!=-1?e=11:t.v?e=t.W?4:t.Z&&!Jt(t.v,61)?3:2:e=t.Z?t.d?8:6:t.d?7:5;break;case 4:e=1;break;case 22:e=0;break;case 21:e=12;break;default:e=11;}t.s!=e&&(t.s=e,W.y1(t._,e));},q.pM=function(t,e){q.iM(t),q.cM(t,tt.TH(e,true,null));},q.qM=function(t,e,A,P){this.V=t,this.U=e,this._=A,this.k=P,this.S=new U.Lj(),this.w=5,this.t=-1,this.u=-1,this.v=null,this.e=-1,this.F=6,this.B=0,this.D=-1,this.C=0,this.A=null,this.b=true,this.a=false,this.X=0,this.s=11,this.i=new T.Pjb(),this.c=331,this.$=1,this.ab=(j1.IN==-1&&(j1.IN=j1.KN()),j1.IN),this.T=Ft(j.Math.round((j1.IN==-1&&(j1.IN=j1.KN()),j1.IN*24))),(this.U&6)!=0&&(this.U|=1),(this.U&12)!=0&&(this.I=new n.ek()),this.db=4;},x(286,1,hn,q.qM),y.lc=function(t){var e;Jt(t,139)?q.tL((e=this,e)):Jt(t,183)?q.uL(this,t):Jt(t,155)&&q.vL(this,t);},y.a=false,y.b=false,y.c=0,y.d=false,y.e=0,y.f=0,y.g=false,y.q=0,y.s=0,y.t=0,y.u=0,y.w=0,y.B=0,y.C=0,y.D=0,y.F=0,y.H=0,y.J=0,y.K=0,y.N=0,y.O=false,y.T=0,y.U=0,y.W=false,y.X=0,y.Y=0,y.Z=false,y.$=0,y.ab=0,y.db=0,y.eb=0,y.gb=0,y.hb=0,y.jb=0,y.kb=0,q.YK=true,l.S9=X(286),q.sM=function(t,e,A){this.a=t,this.c=e,this.b=A;},x(296,1,{262:1},q.sM),y.mc=function(){q.NL(this.a,this.c,this.b);},y.c=0,l.Q9=X(296),q.vM=function(t,e){this.a=t,this.b=e;},x(297,1,{},q.vM),y.Qb=function(t,e){return q.OL(this.a,this.b,t,e);},y.lb=function(t){return this===t;},l.R9=X(297),q.AM=function(){q.AM=o1,q.yM=Ft(j.Math.round(Ft(j.Math.round((j1.IN==-1&&(j1.IN=j1.KN()),j1.IN*2))))),q.zM=(j1.IN==-1&&(j1.IN=j1.KN()),j1.IN*21);},q.BM=function(t,e,A,P,r){var i;P|=P1,i=q.DM(A),P!=-1&&(R.V2(e,P),R.L2(e,i[0],i[1],q.zM,q.zM)),A==20?R.E2(e,t.c,0,t.d*84,84,84,i[0],i[1],q.zM,q.zM):R.E2(e,r?t.g:t.i,(i[0]-q.yM)*t.j,(i[1]-q.yM)*t.j,84,84,i[0],i[1],q.zM,q.zM);},q.CM=function(t,e){var A,P;if(e.j==1){if(A=q.EM(t,e),!(A>=0&&A<34&&(A!=t.b||A==20||A==33)&&(A!=19||(t.a.U&4)!=0)&&(A!=21||(t.a.U&8)!=0)))return;A==20&&A==t.b&&(t.d=++t.d%3),t.k=A,W.f1(t.n);}else if(e.j==2){if(t.k==-1)return;if(P=q.EM(t,e),P!=t.k||t.k==19&&(t.a.U&4)==0||t.k==21&&(t.a.U&8)==0){t.k=-1,W.f1(t.n);return;}if(t.k=-1,P==0||P==1||P==17){W.f1(t.n),q.eL(t.a,P);return;}t.b=P,W.f1(t.n),t.b==20?q.lM(t.a,t.d==0?101:t.d==1?103:102):(t.b==33&&q.gM(t.a,-1),q.lM(t.a,P));}else (e.j==6||e.j==5)&&(A=q.EM(t,e),A==t.k&&(A=-1),A!=t.f&&(t.f=A,W.f1(t.n)));},q.DM=function(t){var e;return e=z(l.b7,jt,6,2,15,1),e[0]=q.zM*(t/17|0)+q.yM,e[1]=q.zM*(t%17)+q.yM,e;},q.EM=function(t,e){var A,P,r;return P=e.g-q.yM,r=e.i-q.yM,P<0||P>=2*q.zM||r<0||r>=17*q.zM?-1:(A=17*Ft(P/q.zM)+Ft(r/q.zM),A>=0&&A<34&&(A!=t.b||A==20||A==33)&&(A!=19||(t.a.U&4)!=0)&&(A!=21||(t.a.U&8)!=0)?A:-1);},q.FM=function(t){t.i=W.r1(t.a._,Gr),j1.PN(t.i),t.g=W.r1(t.a._,Gr),j1.JN(t.g),t.o=2*q.yM+j1.LN(R._2(t.i)/4|0),t.e=2*q.yM+j1.LN(R.Y2(t.i)/4|0),t.c=W.r1(t.a._,HD),t.j=(j1.IN==-1&&(j1.IN=j1.KN()),4/j1.IN),t.b=5,t.k=-1,t.f=-1;},q.GM=function(t,e){var A,P,r,i,D,u,o;A=W.b1(t.n),r=(lt.I3(),(lt.H3[0]*((A&W0)>>16)+lt.H3[1]*((A&J0)>>8)+lt.H3[2]*(A&255))/255<.5),P=r?lt.J3(A,Dn):A&O0|Ft(j.Math.round(sA*((A&W0)>>16)))<<16|Ft(j.Math.round(sA*((A&J0)>>8)))<<8|Ft(j.Math.round(sA*(A&255))),D=r?lt.J3(A,.5):A&O0|Ft(j.Math.round(Tn*((A&W0)>>16)))<<16|Ft(j.Math.round(Tn*((A&J0)>>8)))<<8|Ft(j.Math.round(Tn*(A&255))),o=R._2(t.i),u=R.Y2(t.i),R.E2(e,t.i,0,0,o,u,q.yM,q.yM,o/t.j,u/t.j),i=q.DM(20),R.E2(e,t.c,0,t.d*84,84,84,i[0],i[1],q.zM,q.zM),(t.a.U&4)==0&&q.BM(t,e,19,-1,true),(t.a.U&8)==0&&q.BM(t,e,21,-1,true),q.BM(t,e,t.b,D,false),t.f!=-1&&t.f!=t.k&&q.BM(t,e,t.f,P,false),t.k!=-1&&q.BM(t,e,t.k,7167924,false);},q.HM=function(t,e){q.AM(),this.n=t,this.a=e,q.FM(this);},x(298,1,hn,q.HM),y.lc=function(t){q.CM(this,t);},y.b=0,y.d=0,y.e=0,y.f=0,y.j=0,y.k=0,y.o=0,q.yM=0,q.zM=0,l.T9=X(298),q.MM=function(){q.MM=o1,q.KM=F(k(l.d7,1),H,6,15,[0,2,1,3]),q.LM=F(k(l.Sbb,1),st,2,6,["plain","italic","bold","italics & bold"]),q.JM=F(k(l.Sbb,1),st,2,6,["8","9","10","12","14","18","24","32"]);},q.NM=function(t){var e,A,P,r,i,D,u,o,Q,L,s;for(t.b=R.p2(t.c),i=q.JM,u=0,Q=i.length;u<Q;++u)P=i[u],R.e2(t.b,P);for(R.i2(t.b,true),R.k2(t.b,""+Ft(t.e.c)),t.a=R.p2(t.c),r=q.LM,D=0,o=r.length;D<o;++D)P=r[D],R.e2(t.a,P);for(L=0,A=0;A<q.KM.length;A++)if(t.e.d==q.KM[A]){L=A;break;}R.j2(t.a,L),e=F(k(l.d7,1),H,6,15,[8,-2,4,-2,8]),s=F(k(l.d7,1),H,6,15,[8,-2,4,-2,8,-2,8]),R.u2(t.c,e,s),R.m2(t.c,R.q2(t.c,"Text size:"),1,1),R.m2(t.c,t.b,3,1),R.m2(t.c,R.q2(t.c,"Text style:"),1,3),R.m2(t.c,t.a,3,3),t.d=R.r2(t.c,20,3),R.n3(t.d,t.e.e),R.n2(t.c,t.d,1,5,3,5);},q.OM=function(t,e){var A,P;if(e.j==0){try{A=T.Qhb(R.g2(t.b),P1,St);}catch(r){if(r=H1(r),Jt(r,40)){R.w2(t.c,"Illegal text size.");return;}else throw Ut(r);}P=q.KM[R.f2(t.a)],n.Yw(t.e,R.m3(t.d),A,P);}R.s2(t.c);},q.PM=function(t,e){q.MM(),this.c=W.p1(t,"Edit Text",this),this.e=e,q.NM(this);},x(295,1,hn,q.PM),y.lc=function(t){q.OM(this,t);},l.U9=X(295),Tt.RM=function(t,e,A){Tt.WK.call(this,e,t),this.a=A;},x(139,161,{139:1},Tt.RM),y.a=0,l.V9=X(139),Tt.WM=function(){Tt.WM=o1,Tt.UM=F(k(l.d7,2),S1,7,0,[F(k(l.d7,1),H,6,15,[-1432180054,-1434146134,-1441453406,-1471067516,-1589095916,-2060839854,1420461386,1454022954,1454024874]),F(k(l.d7,1),H,6,15,[-1431677611,-1431764987,-1432092655,-1433403327,-1438646015,-1459616762,-1543499750,-1879031702,1431634346,1073809066,1073814186,1073834666,1431677610]),F(k(l.d7,1),H,6,15,[-1431737686,-1453057622,-1542187926,-1542188954,-1459351471,-1459351487,Kr,1094713345,1077936134,-1879048186,-1543503866,-1543503846,-1459617766,Yr,On,On]),F(k(l.d7,1),H,6,15,[-1431737686,-1453057622,-1542187926,-1542188954,-1459351471,-1459351487,Kr,1094713345,1077952518,-1879031802,-1543154682,-1543487462,-1459601382,Yr,On,On]),F(k(l.d7,1),H,6,15,[BP,BP,-1431741782,-1437330070,-1459351462,-1459351482,Kr,-1858076671,-1874853882,-1879048186,-1543503866,-1543503846,-1459617766,Yr,On,On]),F(k(l.d7,1),H,6,15,[BP,-1432005270,-1454374890,-1543503871,-1879048191,1073741825,1073741846,1073743210,-1795073366,1079421610,1147841194,-1783977302,mP,mP,-1498764630]),F(k(l.d7,1),H,6,15,[BP,-1432005270,-1454374890,-1543438335,-1878982655,1075138561,1073807382,1073808746,-1795073366,1079421610,1147841194,-1783977302,mP,mP,-1498764630]),F(k(l.d7,1),H,6,15,[1431655786,ke,bn,ke,bn,ke,bn,ke,bn,ke,bn,ke,1145324650]),F(k(l.d7,1),H,6,15,[1431655786,ke,bn,ke,bn,ke,1073758314,1073758250,1074091114,1073758250,1073758314,ke,1145324650]),F(k(l.d7,1),H,6,15,[-1432003926,-1437248086,1767548266,1437248090,1437248086,1433053846,1433037477,-1431672155,-1432004955,1433037477,1433037462,1437248086,1437248090,1767548266,-1437248086,-1432003926]),null]),Tt.SM=F(k(l.d7,1),H,6,15,[1,7,7,5,8,7,8,7,8,7,2,14,2,14,0,0,0,0,8,8,0,0]),Tt.TM=F(k(l.d7,1),H,6,15,[4,20,10,29,18,18,18,18,18,18,16,11,16,11,4,5,4,5,16,16,16,16,0,0,0,0,12,2]),Tt.VM=F(k(l.Sbb,1),st,2,6,["chain.png","eraser.png","hand.png","handPlus.png","fist.png","lasso.png","lassoPlus.png","rect.png","rectPlus.png","zoom.png","invisible.png",null,null,"pointingHand.png"]);},Tt.XM=function(t,e){Tt.WM();var A,P,r,i;if(Tt.UM[e]!=null){for(P=0;P<Tt.UM[e].length;P++)for(i=Tt.UM[e][P],r=15;r>=0;r--)(i&3)==0&&R.a3(t,r,P,-1),i>>=2;for(A=0;A<Tt.UM[e].length;A++)for(i=Tt.UM[e][A],r=15;r>=0;r--)(i&3)==1&&R.a3(t,r,A,O0),i>>=2;}},Tt.YM=function(t){n._c(),n.ce.call(this,t);},x(105,122,{122:1,105:1},Tt.YM),y.pb=function(t){R.F2(this.B,t.a,t.c,t.b,t.d);},y.qb=function(t){R.D2(this.B,t.a,t.c,t.b,t.d);},y.rb=function(t){R.K2(this.B,new R.i3(t));},y.sb=function(t,e,A){R.C2(this.B,e,A,t);},y.tb=function(t,e,A){R.J2(this.B,t,e,A);},y.ub=function(t){return R.N2(this.B,t).b;},y.vb=function(){return this.b;},y.wb=function(){n.vd(this),this.a=1;},y.zb=function(t){this.a=t,R.U2(this.B,this.a);},y.Ab=function(t){R.V2(this.B,t);},y.Bb=function(t){this.b=t,this.B&&R.T2(this.B,t,false,false);},y.a=0,y.b=0,l.W9=X(105),Tt.iN=function(t,e){U.rj(t.a,e);},Tt.jN=function(t,e){var A,P;for(P=new U.wqb(t.a);P.a<P.c.a.length;)A=U.vqb(P),A.lc(e);},Tt.kN=function(){this.a=new U.Lj();},x(221,1,{}),l.X9=X(221),Tt.lN=function(t,e,A,P){Tt.WK.call(this,t,P),this.b=e,this.a=A;},x(183,161,{183:1},Tt.lN),y.a=false,y.b=0,l.Z9=X(183),Tt.nN=function(){Tt.nN=o1,Tt.mN=F(k(l.Sbb,1),st,2,6,["none","pressed","released","clicked","entered","exited","moved","dragged"]);},Tt.oN=function(t,e,A,P,r,i,D,u,o,Q){Tt.nN(),Tt.WK.call(this,t,Q),this.b=e,this.c=A,this.g=P,this.i=r,this.f=i,this.d=D,this.a=u,this.e=o;},x(155,161,{155:1},Tt.oN),y.ob=function(){return Tt.mN[this.j]+" x:"+this.g+" y:"+this.i+" button:"+this.b+" clicks:"+this.c+(this.f?" shift":"")+(this.d?" ctrl":"")+(this.a?" alt":"")+" isPopup:"+(this.e?"y":"n");},y.a=false,y.b=0,y.c=0,y.d=false,y.e=false,y.f=false,y.g=0,y.i=0,l.$9=X(155),Tt.qN=function(){},Tt.rN=function(t,e){this.a=t,this.b=e;},x(55,1,{55:1},Tt.qN,Tt.rN),y.ob=function(){return "x:"+this.a+" y:"+this.b;},y.a=0,y.b=0,l._9=X(55),Tt.tN=function(t,e,A){t.a==t.b.length&&(t.b=U.Dqb(t.b,2*t.a),t.c=U.Dqb(t.c,2*t.a)),t.b[t.a]=e,t.c[t.a]=A,++t.a;},Tt.uN=function(t,e){return t.b[e];},Tt.vN=function(t,e){return t.c[e];},Tt.wN=function(t){t.a>0&&--t.a;},Tt.xN=function(){this.b=z(l.b7,jt,6,64,15,1),this.c=z(l.b7,jt,6,64,15,1),this.a=0;},Tt.yN=function(t){this.b=z(l.b7,jt,6,t,15,1),this.c=z(l.b7,jt,6,t,15,1),this.a=0;},x(119,1,{404:1},Tt.xN,Tt.yN),y.Gb=function(t,e){var A,P,r;for(r=false,P=this.a-1,A=0;A<this.a;A++)this.c[A]>e!=this.c[P]>e&&t<(this.b[P]-this.b[A])*(e-this.c[A])/(this.c[P]-this.c[A])+this.b[A]&&(r=!r),P=A;return r;},y.a=0,l.aab=X(119),Tt.AN=function(t,e,A){return e>=t.c&&e<=t.c+t.b&&A>=t.d&&A<=t.d+t.a;},Tt.BN=function(t,e){return Tt.AN(t,e.c,e.d)&&Tt.AN(t,e.c+e.b,e.d+e.a);},Tt.CN=function(t,e,A,P,r){t.c=e,t.d=A,t.b=P,t.a=r;},Tt.DN=function(t,e){var A,P,r,i;return r=j.Math.min(t.c,e.c),i=j.Math.min(t.d,e.d),P=j.Math.max(t.c+t.b,e.c+e.b)-r,A=j.Math.max(t.d+t.a,e.d+e.a)-i,new Tt.FN(r,i,P,A);},Tt.EN=function(){},Tt.FN=function(t,e,A,P){this.c=t,this.d=e,this.b=A,this.a=P;},x(35,1,{35:1,404:1},Tt.EN,Tt.FN),y.Gb=function(t,e){return Tt.AN(this,t,e);},y.ob=function(){return "x:"+this.c+" y:"+this.d+" w:"+this.b+" h:"+this.a;},y.a=0,y.b=0,y.c=0,y.d=0,l.bab=X(35),j1.JN=function(t){var e,A,P,r,i;for(A=lt.K3(new F1.Ffb(238,238,238)),P=We&A.a,r=0;r<R._2(t);r++)for(i=0;i<R.Y2(t);i++)e=R.$2(t,r,i),R.a3(t,r,i,(O0&e)+P);},j1.KN=function(){return globalThis.devicePixelRatio||1;},j1.LN=function(t){return Ft(j.Math.round((j1.IN==-1&&(j1.IN=j1.KN()),j1.IN*t)));},j1.IN=-1,j1.ON=function(){j1.ON=o1,j1.NN=F(k(l.d7,1),H,6,15,[5258420,0]),j1.MN=F(k(l.d7,1),H,6,15,[11837695,14737632]);},j1.PN=function(t){j1.ON();var e;e=jP,(T.ejb(w.Xzb(e).substr(0,18),Nn)||T.ejb(w.Xzb(e).substr(0,17),In)||T.ejb(w.Xzb(e).substr(0,14),xn))&&j1.QN(t,j1.MN);},j1.QN=function(t,e){var A,P,r,i,D;for(i=0;i<R._2(t);i++)for(D=0;D<R.Y2(t);D++)for(A=R.$2(t,i,D),r=A&We,P=0;P<j1.NN.length;P++)if(r==j1.NN[P]){R.a3(t,i,D,(O0&A)+e[P]);break;}},B.RN=function(t){t=t||{};var e=0;return t.createSymmetryRank&&(e|=1),t.considerStereoheterotopicity&&(e|=2),t.encodeAtomCustomLabels&&(e|=8),t.encodeAtomSelection&&(e|=16),t.assignParitiesToTetrahedralN&&(e|=32),t.coordsAre3d&&(e|=64),t.createPseudoStereoGroups&&(e|=128),t.distinguishRacemicOrGroups&&(e|=256),t.tieBreakFreeValenceAtoms&&(e|=512),t.encodeAtomCustomLabelsWithoutRanking&&(e|=i0),t.neglectAnyStereoInformation&&(e|=z1),e;},B.SN=function(t,e){this.a=new n.Ng(t.a,B.RN(e));},x(401,1,{},B.SN),y.getCanMolecule=function(t){var e;return e=n.jg(this.a,t),new B.sT(e);},y.getEncodedCoordinates=function(t){return n.mg(this.a,t);},y.getEncodedMapping=function(){return n.ng(this.a);},y.getFinalRank=function(){return this.a.f;},y.getGraphAtoms=function(){return n.og(this.a);},y.getGraphIndexes=function(){return n.pg(this.a);},y.getIDCode=function(){return n.qg(this.a);},y.getSymmetryRank=function(t){return n.ug(this.a,t);},y.getSymmetryRanks=function(){return this.a.g;},y.hasCIPParityDistinctionProblem=function(){return this.a.d;},y.invalidateCoordinates=function(){this.a.s=null;},y.normalizeEnantiomer=function(){return n.Eg(this.a);},y.setParities=function(){n.Ig(this.a);},l.dab=X(401),B.jO=function(){},B.kO=function(t,e){switch(e){default:case 0:return n.Sh(t.a,0);case 1:return n.Sh(t.a,1);case 2:return n.Sh(t.a,2);case 3:return n.Sh(t.a,3);case 4:return n.Sh(t.a,4);}},x(405,1,{},B.jO),B.eO=2,B.fO=0,B.gO=1,B.hO=4,B.iO=3,l.cab=X(405),B.pO=function(t){B.a$||B.d$(),this.a=new _.TAb(l.Oeb(t));},x(406,1,{},B.pO),y.getConformerCount=function(){return this.a.p;},y.getNextConformerAsMolecule=function(t){var e,A;return e=t?t.a:null,A=_.HAb(this.a,e),A?A==e?t:new B.sT(A):null;},y.getOneConformerAsMolecule=function(t){var e;return e=_.JAb(this.a,t.a),e?t:null;},y.getPotentialConformerCount=function(){return _.KAb(this.a);},y.oc=function(t,e,A,P){return _.NAb(this.a,t.a,e,A,P);},y.initializeConformers=function(t,e){e===void 0&&(e={});var A=e.strategy;A===void 0&&(A=3);var P=e.maxTorsionSets;P===void 0&&(P=uP);var r=e.use60degreeSteps;return r===void 0&&(r=false),this.oc(t,A,P,r);},B.lO=4,B.mO=3,B.nO=1,B.oO=2,l.eab=X(406),B.wO=function(){},B.xO=function(t,e,A,P,r){return nt.iG(t,e,A,P,r);},x(407,1,{},B.wO),l.fab=X(407),B.zO=function(){throw new Error("drug likeness must be assessed first");},B.AO=function(){B.a$||B.d$(),this.a=new nt.nG();},x(408,1,{},B.AO),y.assessDruglikeness=function(t){var e;return nt.lG(this.a,(e=t.a,!N1.r3&&(N1.r3=new N1.s3()),e));},y.getDetail=function(){var t;return t=this.a.a,!t&&B.zO(),N1.w3(t);},y.getDruglikenessString=function(t){return nt.mG(t.a);},B.yO=-999,l.gab=X(408),B.KO=function(t,e,A){B.a$||B.d$(),B.NO(e),this.a=new Y.zC(t.a,e,new U.utb());},B.NO=function(t){T.ejb(t,Jr)&&!B.HO?(Y.tC(),Y.AC(Jr,new Y.rD("/resources/forcefield/mmff94/outofplane.csv","/resources/forcefield/mmff94/torsion.csv")),B.HO=true):T.ejb(t,Zr)&&!B.IO?(Y.tC(),Y.AC(Zr,new Y.rD(MD,"/resources/forcefield/mmff94/94s/torsion.csv")),B.IO=true):T.ejb(t,sP)&&!B.JO&&(Y.tC(),Y.AC(sP,new Y.rD(MD,"/resources/forcefield/mmff94/94s/torsionPlus.csv")),B.JO=true);},x(409,1,{},B.KO),y._minimise=function(t,e,A){return we.yB(this.a,t,e);},y.getTotalEnergy=function(){return Y.vC(this.a);},y.size=function(){return this.a.b.q;},B.EO=Jr,B.FO=Zr,B.GO=sP,B.HO=false,B.IO=false,B.JO=false,l.hab=X(409),B.mT=function(){B.mT=o1,B.lT=new Tt.EN(),B.tP=F(k(l.Sbb,1),st,2,6,["?","H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr","Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"R4","R5","R6","R7","R8","R9","R10","R11","R12","R13","R14","R15","R16","R1","R2","R3","A","A1","A2","A3",R1,R1,"D","T","X","R","H2","H+","Nnn","HYD","Pol",R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,R1,"Ala","Arg","Asn","Asp","Cys","Gln","Glu","Gly","His","Ile","Leu","Lys","Met","Phe","Pro","Ser","Thr","Trp","Tyr","Val"]),B.kT=F(k(l.yeb,1),Kt,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]),B.eR=F(k(l._6,2),Te,10,0,[null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[0]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[0]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[2,4,6]),F(k(l._6,1),J,6,15,[1,3,5,7]),F(k(l._6,1),J,6,15,[0]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,null,null,null,null,null,null,null,null,F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[2,4,6]),F(k(l._6,1),J,6,15,[1,3,5,7]),F(k(l._6,1),J,6,15,[0,2]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,null,null,null,null,null,null,null,null,F(k(l._6,1),J,6,15,[1,2,3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[2,4,6]),F(k(l._6,1),J,6,15,[1,3,5,7]),F(k(l._6,1),J,6,15,[0,2,4,6]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[2])]),B.xS=F(k(l._6,2),Te,10,0,[null,F(k(l._6,1),J,6,15,[1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),null,null,F(k(l._6,1),J,6,15,[-3]),F(k(l._6,1),J,6,15,[-2]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),null,F(k(l._6,1),J,6,15,[-3]),F(k(l._6,1),J,6,15,[-2]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3,4]),F(k(l._6,1),J,6,15,[2,3,4,5]),F(k(l._6,1),J,6,15,[2,3,6]),F(k(l._6,1),J,6,15,[2,3,4,7]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[1,2]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[-3,3,5]),F(k(l._6,1),J,6,15,[-2]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[6]),F(k(l._6,1),J,6,15,[4,6,7]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[-3,3,5]),F(k(l._6,1),J,6,15,[-2,4,6]),F(k(l._6,1),J,6,15,[-1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[5]),F(k(l._6,1),J,6,15,[6]),F(k(l._6,1),J,6,15,[4,6,7]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[1,3]),F(k(l._6,1),J,6,15,[1,2]),F(k(l._6,1),J,6,15,[1,3]),F(k(l._6,1),J,6,15,[2,4]),F(k(l._6,1),J,6,15,[3,5]),F(k(l._6,1),J,6,15,[-2,2,4]),F(k(l._6,1),J,6,15,[-1,1]),null,F(k(l._6,1),J,6,15,[1]),F(k(l._6,1),J,6,15,[2]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[4]),F(k(l._6,1),J,6,15,[4,5]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3,4,5,6]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3,4]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[2,3]),F(k(l._6,1),J,6,15,[3])]),B.dP=F(k(l.Sbb,1),st,2,6,[Tr,br,cr]);},B.nT=function(t,e,A,P){return {x:t,y:e,width:A,height:P};},B.oT=function(t){return t.a;},B.pT=function(t,e,A,P){var r;return r=0,e&&(r|=1),A&&(r|=2),P&&(r|=4),n.pr(new n.xr(t.a,r));},B.qT=function(){B.mT(),B.rT.call(this,32,32,null);},B.rT=function(t,e,A){B.mT(),A?this.a=A:this.a=new n.tw(t,e);},B.sT=function(t){B.mT(),B.rT.call(this,0,0,t);},B.WT=function(t,e){return {molecule:t,map:e};},B.kU=function(t,e){B.mT();var A;return typeof e===Xn&&(e=true),typeof e===u0?A=B.mU(t,e):typeof e===M0&&(A=B.lU(t,e)),A;},B.lU=function(t,e){return new B.sT(n.Pq(new n.$q(false),t,e));},B.mU=function(t,e){return new B.sT(n.Oq(new n.$q(e),t));},B.nU=function(t){return B.mT(),new B.sT(n.es(new n.Bs(),t));},B.oU=function(t){B.mT();var e,A,P;return P=new n.Cs(),A=(P.j=null,n.zs(P,new Rt.igb(new Rt.tgb(t)))?P.j:null),e=P.f==null?n.kp(P.j):P.f,B.WT(new B.sT(A),e);},B.pU=function(t,e){B.mT(),e=e||{};var A=!e.noCoordinates,P=!e.noStereo;return B.qU(t,A,P);},B.qU=function(t,e,A){var P,r,i;return i=new n.Hv(),r=new B.qT(),n.xv(i,r.a,w.xzb((P=t,w.szb(),P)),e,A),r;},B.yU=function(t,e,A,P){return B.mT(),n.mo(t,e,A,P);},B.zU=function(t,e){return B.mT(),n.no(t,e);},B.ZU=function(t,e){return B.mT(),n.oo(t,e);},B.CV=function(){return B.mT(),n.Qk(),n.Qk(),n.Pk;},B.IW=function(t){return B.mT(),n.po(t);},B.JW=function(t){return B.mT(),n.qo(t);},B.fY=function(t){B.mT(),n.Qk(),n.Pk=t;},x(42,1,{42:1},B.qT,B.rT,B.sT),y.addAtom=function(t){return n.Tk(this.a,t);},y.addBond=function(t,e){return n.Uk(this.a,t,e);},y.addFragment=function(t,e,A){n.wo(this.a,t.a,e,A);},y.pc=function(){Yt.Kz(this.a);},y.qc=function(t){Yt.Lz(this.a,t);},y.addImplicitHydrogens=function(t){t===void 0?this.pc():this.qc(t);},y.rc=function(){Yt.rz(),Yt.sz(this.a,1);},y.sc=function(t){Yt.sz(this.a,t);},y.addMissingChirality=function(t){t===void 0?this.rc():this.sc(t);},y.addMolecule=function(t){return n.Xk(this.a,t.a);},y.addOrChangeAtom=function(t,e,A,P,r,i,D){return n.Zk(this.a,t,e,A,P,r,i,D);},y.addOrChangeBond=function(t,e,A){return n.$k(this.a,t,e,A);},y.addRing=function(t,e,A,P,r){return n._k(this.a,t,e,A,P,r);},y.addRingToAtom=function(t,e,A,P){return n.al(this.a,t,e,A,P);},y.addRingToBond=function(t,e,A,P){return n.bl(this.a,t,e,A,P);},y.addSubstituent=function(t,e){return n.cl(this.a,t.a,e);},y.calculateTorsion=function(t){return n.dl(this.a,t);},y.canonizeCharge=function(t){return n.yo(this.a,t,false);},y.changeAtom=function(t,e,A,P,r){return n.el(this.a,t,e,A,P,r);},y.changeAtomCharge=function(t,e){return n.gl(this.a,t,e);},y.changeBond=function(t,e){return n.hl(this.a,t,e);},y.clear=function(){n.$v(this.a);},y.convertStereoBondsToSingleBonds=function(t){n.Ao(this.a,t);},y.copyAtom=function(t,e,A,P){return n.kl(this.a,t.a,e,A,P);},y.copyBond=function(t,e,A,P,r,i){return n.ml(this.a,t.a,e,A,P,r,i);},y.copyMolecule=function(t){n.nl(this.a,t.a);},y.copyMoleculeByAtoms=function(t,e,A,P){n.Bo(this.a,t.a,e,A,P);},y.copyMoleculeByBonds=function(t,e,A,P){return n.Co(this.a,t.a,e,A,P);},y.copyMoleculeProperties=function(t){n._v(this.a,t.a);},y.deleteAtom=function(t){n.pl(this.a,t);},y.deleteAtomOrBond=function(t,e){return n.ql(this.a,t,e);},y.deleteAtoms=function(t){return n.rl(this.a,t);},y.deleteBond=function(t){n.sl(this.a,t);},y.deleteBondAndSurrounding=function(t){n.tl(this.a,t);},y.deleteMarkedAtomsAndBonds=function(){return n.ul(this.a);},y.deleteMolecule=function(){n.$v(this.a);},y.deleteSelectedAtoms=function(){return n.vl(this.a);},y.ensureHelperArrays=function(t){n.aw(this.a,t);},y.findAlleneCenterAtom=function(t){return n.Eo(this.a,t);},y.findAlleneEndAtom=function(t,e){return n.Fo(this.a,t,e);},y.findAtom=function(t,e){return n.wl(this.a,t,e);},y.findBINAPChiralityBond=function(t){return n.Go(this.a,t);},y.findBond=function(t,e){return n.xl(this.a,t,e);},y.findRingSystem=function(t,e,A,P){n.Ho(this.a,t,e,A,P);},y.getAbsoluteAtomParity=function(t){return n.bw(this.a,t);},y.getAbsoluteBondParity=function(t){return n.cw(this.a,t);},y.getAllAtoms=function(){return this.a.q;},y.getAllBonds=function(){return this.a.r;},y.getAllConnAtoms=function(t){return n.Ko(this.a,t);},y.getAllConnAtomsPlusMetalBonds=function(t){return n.Lo(this.a,t);},y.getAllHydrogens=function(t){return n.Mo(this.a,t);},y.getAromaticRingCount=function(){return n.No(this.a);},y.getAtomAbnormalValence=function(t){return n.yl(this.a,t);},y.getAtomCIPParity=function(t){return n.zl(this.a,t);},y.getAtomCharge=function(t){return n.Al(this.a,t);},y.getAtomColor=function(t){return n.Bl(this.a,t);},y.getAtomCustomLabel=function(t){return n.Dl(this.a,t);},y.getAtomESRGroup=function(t){return n.Fl(this.a,t);},y.getAtomESRType=function(t){return n.Gl(this.a,t);},y.getAtomLabel=function(t){return n.Hl(this.a,t);},y.getAtomList=function(t){return n.Il(this.a,t);},y.getAtomListString=function(t){return n.Jl(this.a,t);},y.getAtomMapNo=function(t){return n.Kl(this.a,t);},y.getAtomMass=function(t){return n.Ll(this.a,t);},y.getAtomParity=function(t){return n.Ml(this.a,t);},y.getAtomPi=function(t){return n.Po(this.a,t);},y.getAtomPreferredStereoBond=function(t){return n.Qo(this.a,t);},y.getAtomQueryFeaturesObject=function(t){return N1.p3(this.a,t);},y.getAtomRadical=function(t){return n.Ol(this.a,t);},y.getAtomRingBondCount=function(t){return n.Ro(this.a,t);},y.getAtomRingCount=function(t,e){return n.So(this.a,t,e);},y.getAtomRingSize=function(t){return n.To(this.a,t);},y.getAtomX=function(t){return n.Pl(this.a,t);},y.getAtomY=function(t){return n.Ql(this.a,t);},y.getAtomZ=function(t){return n.Rl(this.a,t);},y.getAtomicNo=function(t){return n.Sl(this.a,t);},y.getAtoms=function(){return this.a.f;},y.getAverageBondLength=function(t){return n.Uo(this.a,t);},y.getAverageTopologicalAtomDistance=function(){return n.Vo(this.a);},y.getBond=function(t,e){return n.Wo(this.a,t,e);},y.getBondAngle=function(t,e){return n.Xl(this.a,t,e);},y.getBondAtom=function(t,e){return n.Yl(this.a,t,e);},y.getBondBridgeMaxSize=function(t){return n.Zl(this.a,t);},y.getBondBridgeMinSize=function(t){return n.$l(this.a,t);},y.getBondCIPParity=function(t){return n._l(this.a,t);},y.getBondESRGroup=function(t){return n.am(this.a,t);},y.getBondESRType=function(t){return n.bm(this.a,t);},y.getBondLength=function(t){return n.cm(this.a,t);},y.getBondOrder=function(t){return n.em(this.a,t);},y.getBondParity=function(t){return n.fm(this.a,t);},y.getBondPreferredStereoBond=function(t){return n.nq(this.a,t);},y.getBondQueryFeatures=function(t){return n.gm(this.a,t);},y.getBondQueryFeaturesObject=function(t){return N1.q3(this.a,t);},y.getBondRingSize=function(t){return n.Xo(this.a,t);},y.getBondType=function(t){return n.hm(this.a,t);},y.getBondTypeSimple=function(t){return n.im(this.a,t);},y.getBonds=function(){return this.a.g;},y.getBounds=function(){var t;return t=n.jm(this.a,B.lT),t?B.nT(t.c,t.d,t.b,t.a):null;},y.getCanonizedIDCode=function(t){var e;return e=new n.Ng(this.a,t),n.qg(e);},y.getChiralText=function(){return n.dw(this.a);},y.getChirality=function(){return this.a.I;},y.getCompactCopy=function(){return new B.sT(n.ew(this.a));},y.getConnAtom=function(t,e){return n.Yo(this.a,t,e);},y.getConnAtoms=function(t){return n.Zo(this.a,t);},y.getConnBond=function(t,e){return n.$o(this.a,t,e);},y.getConnBondOrder=function(t,e){return n._o(this.a,t,e);},y.getDefaultMaxValenceUncharged=function(t){return n.km(this.a,t);},y.getDiastereotopicAtomIDs=function(){return Yt.tz(this.a);},y.getElectronValenceCorrection=function(t,e){return n.lm(this.a,t,e);},y.getExcludedNeighbourCount=function(t){return n.ap(this.a,t);},y.getExplicitHydrogens=function(t){return n.bp(this.a,t);},y.getFinalRanks=function(t){var e;return e=new n.Ng(this.a,t),e.f;},y.getFisherProjectionParity=function(t,e,A,P){return n.dp(this.a,t,e,A,P);},y.getFragmentAtoms=function(t,e){return n.ep(this.a,t,e);},y.getFragmentNumbers=function(t,e,A){return n.gp(this.a,t,e,A);},y.getFragments=function(){var t,e,A;for(t=n.gw(this.a),A=z(l.jab,_r,42,t.length,0,1),e=0;e<t.length;e++)A[e]=new B.sT(t[e]);return A;},y.getFreeValence=function(t){return n.ip(this.a,t);},y.getHandleHydrogenMap=function(){return n.kp(this.a);},y.getHelperArrayStatus=function(){return this.a.T;},y.getHoseCodes=function(t){t=t||{};var e=(typeof t.maxSphereSize===Xn?5:t.maxSphereSize)|0,A=(typeof t.type===Xn?0:t.type)|0;return Yt.vz(this.a,e,A);},y.getIDCode=function(){var t;return t=n.iw(this.a),t==null&&(t=n.qg(new n.Mg(this.a))),t;},y.getIDCodeAndCoordinates=function(){return {idCode:this.getIDCode(),coordinates:this.getIDCoordinates()};},y.getIDCoordinates=function(){var t;return t=n.jw(this.a),t==null&&(t=n.lg(new n.Mg(this.a))),t;},y.getImplicitHigherValence=function(t,e){return n.lp(this.a,t,e);},y.getImplicitHydrogens=function(t){return n.np(this.a,t);},y.getIndex=function(){return N1.x3(this.a);},y.getLowestFreeValence=function(t){return n.op(this.a,t);},y.getMaxAtoms=function(){return this.a.M;},y.getMaxBonds=function(){return this.a.N;},y.getMaxValence=function(t){return n.nm(this.a,t);},y.getMaxValenceUncharged=function(t){return n.om(this.a,t);},y.getMetalBondedConnAtoms=function(t){return n.qp(this.a,t);},y.getMolecularFormula=function(){return new B.E_(this.a);},y.getMoleculeColor=function(){return this.a.O;},y.getMolweight=function(){return n.rp(this.a);},y.getName=function(){return this.a.P;},y.getNonHydrogenNeighbourCount=function(t){return n.sp(this.a,t);},y.getNumberOfHydrogens=function(){return Yt.Mz(this.a);},y.getOccupiedValence=function(t){return n.up(this.a,t);},y.getPath=function(t,e,A,P,r){return n.xp(this.a,t,e,A,P,null,r);},y.getPathBonds=function(t,e,A){n.yp(this.a,t,e,A);},y.getPathLength=function(t,e){return n.zp(this.a,t,e);},y.getRingSet=function(){return new B.j$(n.Dp(this.a));},y.getRotatableBondCount=function(){return n.Ep(this.a);},y.tc=function(t,e,A,P,r,i,D){var u,o,Q,L,s,h,b,c,g,d,m;for(Q=true,L=0;L<this.a.q-1;L++)if(n.Pl(this.a,L)!=n.Pl(this.a,L+1)||n.Ql(this.a,L)!=n.Ql(this.a,L+1)){Q=false;break;}return h=Q?n.ew(this.a):this.a,Q&&Z.iA(new Z.BA(0),h),s=N1.y3(D),o=new n.Iu(h,s,i),o.J=A,n.be(o,null,new Tt.FN(0,0,t,e),q1),u=o.v,n.Td(o,null),m=n.Gu(o),P?(c=Gt(w1(j.Math.round(u.b+r*2))),b=Gt(w1(j.Math.round(u.a+r*2))),g=Gt(w1(j.Math.round(u.c-r))),d=Gt(w1(j.Math.round(u.d-r))),T.rjb(m,'width="\\d+px" height="\\d+px" viewBox="0 0 \\d+ \\d+"','width="'+c+'px" height="'+b+'px" viewBox="'+g+" "+d+" "+c+" "+b+'"')):m;},y.getStereoBond=function(t){return n.Hp(this.a,t);},y.getStereoCenterCount=function(){return n.kw(this.a);},y.uc=function(){return B.oT(this);},y.getStereoProblem=function(t){return n.qm(this.a,t);},y.getSubstituent=function(t,e,A,P,r){return n.Jp(this.a,t,e,A,P.a,r);},y.getSubstituentSize=function(t,e){return n.Kp(this.a,t,e);},y.getSymmetryRank=function(t){return n.lw(this.a,t);},y.getZNeighbour=function(t,e){return n.Lp(this.a,t,e);},y.invalidateHelperArrays=function(t){n.tm(this.a,t);},y.inventCoordinates=function(){var t;t=new Z.AA(),t.g=new U.zvb(0),Z.iA(t,this.a),n.xq(this.a);},y.is3D=function(){return n.um(this.a);},y.isAlkaliMetal=function(t){return n.Pp(this.a,t);},y.isAllylicAtom=function(t){return n.Qp(this.a,t);},y.isAmideTypeBond=function(t){return n.Rp(this.a,t);},y.isAromaticAtom=function(t){return n.Sp(this.a,t);},y.isAromaticBond=function(t){return n.Tp(this.a,t);},y.isAtomConfigurationUnknown=function(t){return n.vm(this.a,t);},y.isAtomMarkedForDeletion=function(t){return n.wm(this.a,t);},y.isAtomParityPseudo=function(t){return n.xm(this.a,t);},y.isAtomStereoCenter=function(t){return n.ym(this.a,t);},y.isAutoMappedAtom=function(t){return n.zm(this.a,t);},y.isBINAPChiralityBond=function(t){return n.Up(this.a,t);},y.isBondBackgroundHilited=function(t){return n.Am(this.a,t);},y.isBondBridge=function(t){return n.Bm(this.a,t);},y.isBondForegroundHilited=function(t){return n.Cm(this.a,t);},y.isBondMarkedForDeletion=function(t){return n.Dm(this.a,t);},y.isBondParityPseudo=function(t){return n.Em(this.a,t);},y.isBondParityUnknownOrNone=function(t){return n.Fm(this.a,t);},y.isChalcogene=function(t){return n.Wp(this.a,t);},y.isDelocalizedBond=function(t){return n.Xp(this.a,t);},y.isEarthAlkaliMetal=function(t){return n.Yp(this.a,t);},y.isElectronegative=function(t){return n.Gm(this.a,t);},y.isElectropositive=function(t){return n.Hm(this.a,t);},y.isFlatNitrogen=function(t){return n.Zp(this.a,t,true);},y.isFragment=function(){return this.a.K;},y.isHalogene=function(t){return n.$p(this.a,t);},y.isMarkedAtom=function(t){return n.Jm(this.a,t);},y.isMetalAtom=function(t){return n.Km(this.a,t);},y.isNaturalAbundance=function(t){return n.Lm(this.a,t);},y.isNitrogenFamily=function(t){return n.aq(this.a,t);},y.isOrganicAtom=function(t){return n.Mm(this.a,t);},y.isPseudoRotatableBond=function(t){return n.bq(this.a,t);},y.isPurelyOrganic=function(){return n.Nm(this.a);},y.isRingAtom=function(t){return n.eq(this.a,t);},y.isRingBond=function(t){return n.fq(this.a,t);},y.isSelectedAtom=function(t){return n.Om(this.a,t);},y.isSelectedBond=function(t){return n.Pm(this.a,t);},y.isSimpleHydrogen=function(t){return n.gq(this.a,t);},y.isSmallRingAtom=function(t){return n.hq(this.a,t);},y.isSmallRingBond=function(t){return n.iq(this.a,t);},y.isStabilizedAtom=function(t){return n.jq(this.a,t);},y.isStereoBond=function(t){return n.Qm(this.a,t);},y.markAtomForDeletion=function(t){n.Tm(this.a,t);},y.markBondForDeletion=function(t){n.Um(this.a,t);},y.normalizeAmbiguousBonds=function(){return n.kq(this.a);},y.removeAtomColors=function(){n.Xm(this.a);},y.removeAtomCustomLabels=function(){this.a.t=null;},y.removeAtomMarkers=function(){n.Ym(this.a);},y.removeAtomSelection=function(){n.Zm(this.a);},y.removeBondHiliting=function(){n.$m(this.a);},y.removeExplicitHydrogens=function(){n.pq(this.a,true);},y.removeQueryFeatures=function(){return n.an(this.a);},y.renumberESRGroups=function(t){return n.bn(this.a,t);},y.scaleCoords=function(t){n.cn(this.a,t);},y.setAllAtoms=function(t){n.dn(this.a,t);},y.setAllBonds=function(t){n.en(this.a,t);},y.setAssignParitiesToNitrogen=function(t){n.mw(this.a,t);},y.setAtomAbnormalValence=function(t,e){n.fn(this.a,t,e);},y.setAtomCIPParity=function(t,e){n.gn(this.a,t,e);},y.setAtomCharge=function(t,e){n.hn(this.a,t,e);},y.setAtomColor=function(t,e){n.jn(this.a,t,e);},y.setAtomConfigurationUnknown=function(t,e){n.kn(this.a,t,e);},y.setAtomCustomLabel=function(t,e){n.ln(this.a,t,e);},y.setAtomESR=function(t,e,A){n.nn(this.a,t,e,A);},y.setAtomList=function(t,e,A){n.pn(this.a,t,e,A);},y.setAtomMapNo=function(t,e,A){n.qn(this.a,t,e,A);},y.setAtomMarker=function(t,e){n.rn(this.a,t,e);},y.setAtomMass=function(t,e){n.sn(this.a,t,e);},y.setAtomParity=function(t,e,A){n.tn(this.a,t,e,A);},y.setAtomQueryFeature=function(t,e,A){n.un(this.a,t,e,A);},y.setAtomRadical=function(t,e){n.vn(this.a,t,e);},y.setAtomSelection=function(t,e){n.wn(this.a,t,e);},y.setAtomX=function(t,e){n.yn(this.a,t,e);},y.setAtomY=function(t,e){n.zn(this.a,t,e);},y.setAtomZ=function(t,e){n.An(this.a,t,e);},y.setAtomicNo=function(t,e){n.Bn(this.a,t,e);},y.setBondAtom=function(t,e,A){n.Cn(this.a,t,e,A);},y.setBondBackgroundHiliting=function(t,e){n.Dn(this.a,t,e);},y.setBondCIPParity=function(t,e){n.En(this.a,t,e);},y.setBondESR=function(t,e,A){n.Fn(this.a,t,e,A);},y.setBondForegroundHiliting=function(t,e){n.Gn(this.a,t,e);},y.setBondOrder=function(t,e){n.Hn(this.a,t,e);},y.setBondParity=function(t,e,A){n.In(this.a,t,e,A);},y.setBondParityUnknownOrNone=function(t){n.Jn(this.a,t);},y.setBondQueryFeature=function(t,e,A){n.Kn(this.a,t,e,A);},y.setBondType=function(t,e){n.Ln(this.a,t,e);},y.setChirality=function(t){n.Mn(this.a,t);},y.setFragment=function(t){n.Nn(this.a,t);},y.setHelperValidity=function(t){n.On(this.a,t);},y.setHydrogenProtection=function(t){n.Pn(this.a,t);},y.setMaxAtoms=function(t){n.Qn(this.a,t);},y.setMaxBonds=function(t){n.Rn(this.a,t);},y.setMoleculeColor=function(t){n.Sn(this.a,t);},y.setName=function(t){n.Tn(this.a,t);},y.setParitiesValid=function(t){n.uq(this.a,t);},y.setStereoBondFromAtomParity=function(t){n.vq(this.a,t);},y.setStereoBondFromBondParity=function(t){n.wq(this.a,t);},y.setStereoBondsFromParity=function(){n.xq(this.a);},y.setToRacemate=function(){this.a.L=true;},y.setUnknownParitiesToExplicitlyUnknown=function(){n.nw(this.a);},y.shareSameFragment=function(t,e){return n.zp(this.a,t,e)!=-1;},y.stripIsotopInfo=function(){return n.Wn(this.a);},y.stripSmallFragments=function(t){return n.yq(this.a,t);},y.stripStereoInformation=function(){n.ow(this.a);},y.suggestBondType=function(t,e){return n.Xn(this.a,t,e);},y.supportsImplicitHydrogen=function(t){return n.zq(this.a,t);},y.swapAtoms=function(t,e){n.Yn(this.a,t,e);},y.swapBonds=function(t,e){n.Zn(this.a,t,e);},y.toIsomericSmiles=function(t){t=t||{};var e=t.createSmarts===true,A=t.includeMapping===true,P=t.kekulizedOutput===true;return this.vc(e,A,P);},y.vc=function(t,e,A){return B.pT(this,t,e,A);},y.toMolfile=function(){var t;return t=new n.$r(this.a),t.a.a;},y.toMolfileV3=function(){var t;return t=new n.Is(this.a),t.b.a;},y.toSVG=function(t,e,A,P){if(typeof t!==$1||typeof e!==$1)throw new Error("Molecule#toSVG requires width and height to be specified");P=P||{};var r=P.factorTextSize||1,i=P.autoCrop===true,D=typeof P.autoCropMargin===Xn?5:P.autoCropMargin,u=this.tc(t,e,r,i,D,A,P);return u=u.replace("<style>","<style> text {font-family: sans-serif;}"),P.fontWeight&&(u=u.replace(/font-size=/g,'font-weight="'+P.fontWeight+'" font-size=')),P.strokeWidth&&(u=u.replace(/stroke-width="[^"]+"/g,fP+P.strokeWidth+'"')),u;},y.toSmarts=function(){return n.pr(new n.xr(this.a,1));},y.toSmiles=function(){return B.pT(this,false,false,true);},y.translateCoords=function(t,e){n.$n(this.a,t,e);},y.validate=function(){n.qw(this.a);},y.zoomAndRotate=function(t,e,A){n.ao(this.a,t,e,A);},y.zoomAndRotateInit=function(t,e){n.bo(this.a,t,e);},B.PO=32,B.QO=2,B.RO=4,B.SO=6,B.TO=64,B.UO=128,B.VO=1,B.WO=256,B.XO=8,B.YO=i0,B.ZO=16,B.$O=z1,B._O=512,B.aP=SD,B.bP=24,B.cP=SD,B.eP=cr,B.fP=Tr,B.gP=br,B.hP=0,B.iP=3,B.jP=1,B.kP=2,B.lP=64,B.mP=384,B.nP=448,B.oP=192,B.pP=256,B.qP=0,B.rP=320,B.sP=128,B.uP=1,B.vP=2,B.wP=4,B.xP=0,B.yP=3,B.zP=1,B.AP=ZA,B.BP=2,B.CP=1,B.DP=2,B.EP=je,B.FP=3,B.GP=25,B.HP=5,B.IP=39,B.JP=Ln,B.KP=r0,B.LP=Ie,B.MP=AA,B.NP=Z0,B.OP=4,B.PP=7,B.QP=nA,B.RP=eA,B.SP=Q0,B.TP=Y1,B.UP={l:4194302,m:4193343,h:7},B.VP=5,B.WP=17,B.XP=ae,B.YP=Me,B.ZP=7,B.$P=32,B._P=z1,B.aQ=46,B.bQ=Cn,B.cQ=128,B.dQ=A0,B.eQ=x0,B.fQ=n9,B.gQ=256,B.hQ=Ue,B.iQ=Oe,B.jQ=A9,B.kQ=512,B.lQ=ye,B.mQ=q1,B.nQ=16,B.oQ=P9,B.pQ=i0,B.qQ=se,B.rQ=32,B.sQ=fA,B.tQ=F0,B.uQ=64,B.vQ=4,B.wQ=8,B.xQ=q0,B.yQ=lA,B.zQ=134217728,B.AQ=3,B.BQ=14,B.CQ=$e,B.DQ=Qe,B.EQ=Se,B.FQ=qe,B.GQ=Ve,B.HQ=Ee,B.IQ=Ne,B.JQ=un,B.KQ=120,B.LQ=4,B.MQ=3,B.NQ=2,B.OQ=he,B.PQ=dA,B.QQ=he,B.RQ=E0,B.SQ=30,B.TQ=Ur,B.UQ=Be,B.VQ=3,B.WQ=22,B.XQ=Dr,B.YQ=2,B.ZQ=44,B.$Q=48,B._Q=32,B.aR=0,B.bR=16,B.cR=4,B.dR=48,B.fR=1,B.gR=0,B.hR=3,B.iR=2,B.jR=1,B.kR=0,B.lR=3,B.mR=2,B.nR=We,B.oR=oA,B.pR=2,B.qR=21,B.rR=F0,B.sR=31,B.tR=5,B.uR=0,B.vR=Le,B.wR=8,B.xR=7680,B.yR=4,B.zR=9,