/*
 * openchemlib-utils v8.8.1
 * Various utilities that extends openchemlib-js like HOSE codes or diastereotopic IDs
 * https://github.com/cheminfo/openchemlib-utils#readme
 *
 * Licensed under the MIT license.
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.OCLUtils = {}));
})(this, (function (exports) { 'use strict';

    var _documentCurrentScript = typeof document !== 'undefined' ? document.currentScript : null;
    let xAtomicNumber = 0;
    /**
     * Returns the atomic number of the X atom
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @returns
     */
    function getXAtomicNumber(molecule) {
      if (!xAtomicNumber) {
        const OCL = molecule.getOCL();
        xAtomicNumber = OCL.Molecule.getAtomicNoFromLabel('X', OCL.Molecule.cPseudoAtomX);
      }
      return xAtomicNumber;
    }

    /**
     * Check if a specific atom is a sp3 carbon
     * @param {import('openchemlib').Molecule} molecule
     * @param {number} atomID
     */
    function isCsp3(molecule, atomID) {
      if (molecule.getAtomicNo(atomID) !== 6) return false;
      if (molecule.getAtomCharge(atomID) !== 0) return false;
      if (molecule.getImplicitHydrogens(atomID) + molecule.getConnAtoms(atomID) !== 4) {
        return false;
      }
      return true;
    }

    /**
     * This method put all the chiral centers in the molecule in the same group and racemic
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @param {object} [options={}]
     * @param {object} [options.OCL] - openchemlib library
     */
    function makeRacemic(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      // if we don't calculate this we have 2 epimers
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
      // we need to make one group "AND" for chiral (to force to racemic, this means diastereotopic and not enantiotopic)
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (molecule.getAtomParity(i) !== Molecule.cAtomParityNone) {
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0); // changed to group 0; TLS 9.Nov.2015
        }
      }
      // after the change we need to recalculate the CIP
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
    }

    /**
     * Tag an atom to be able to visualize it
     * @param molecule
     * @param iAtom
     */
    function tagAtom(molecule, iAtom) {
      const customLabel = `${molecule.getAtomLabel(iAtom)}*`;
      molecule.setAtomCustomLabel(iAtom, customLabel);
      if (molecule.getAtomicNo(iAtom) === 1) {
        molecule.setAtomicNo(iAtom, getXAtomicNumber(molecule));
      } else {
        // we can not use X because we would have problems with valencies if it is
        // expanded hydrogens or not
        // we can not only use a custom label because it does not count for the canonisation
        molecule.setAtomMass(iAtom, molecule.getAtomMass(iAtom) + 5);
      }
      return customLabel;
    }

    const FULL_HOSE_CODE$1 = 1;
    const HOSE_CODE_CUT_C_SP3_SP3$1 = 2;
    /**
     * Returns an array of hose code fragments for the specified molecule.
     * @param molecule - The OCL molecule to process.
     * @param options - Options for generating hose codes.
     * @returns An array of hose code fragments.
     */
    function getHoseCodesForAtomsAsFragments(molecule, options = {}) {
      const OCL = molecule.getOCL();
      const {
        allowedCustomLabels,
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE$1,
        tagAtoms = [],
        tagAtomFct = tagAtom
      } = options;
      const rootAtoms = options.rootAtoms ? options.rootAtoms.slice() : [];
      molecule = molecule.getCompactCopy();
      if (tagAtoms.length > 0) {
        internalTagAtoms(molecule, tagAtoms, rootAtoms, tagAtomFct);
      } else {
        // this force reordering of atoms in order to have hydrogens at the end
        molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      }
      if (rootAtoms.length === 0) {
        for (let j = 0; j < molecule.getAllAtoms(); j++) {
          if (allowedCustomLabels?.includes(molecule.getAtomCustomLabel(j) ?? '') || molecule.getAtomCustomLabel(j)) {
            rootAtoms.push(j);
          }
        }
      }
      const fragments = [];
      // keep track of the atoms when creating the fragment
      const mappings = [];
      let min = 0;
      let max = 0;
      const atomMask = new Array(molecule.getAllAtoms()).fill(false);
      const atomList = new Array(molecule.getAllAtoms());
      for (let sphere = 0; sphere <= maxSphereSize; sphere++) {
        if (max === 0) {
          for (const rootAtom of rootAtoms) {
            atomList[max] = rootAtom;
            atomMask[rootAtom] = true;
            max++;
          }
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            const atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              const connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                switch (kind) {
                  case FULL_HOSE_CODE$1:
                    atomMask[connAtom] = true;
                    atomList[newMax++] = connAtom;
                    break;
                  case HOSE_CODE_CUT_C_SP3_SP3$1:
                    if (!(isCsp3(molecule, atom) && isCsp3(molecule, connAtom))) {
                      atomMask[connAtom] = true;
                      atomList[newMax++] = connAtom;
                    }
                    break;
                  default:
                    throw new Error('getHoseCoesForAtom unknown kind');
                }
              }
            }
          }
          min = max;
          max = newMax;
        }
        if (sphere >= minSphereSize) {
          const fragment = new OCL.Molecule(0, 0);
          molecule.copyMoleculeByAtoms(fragment, atomMask, true, mappings);
          // we using atomMapNo field in order to keep track of the original atom number even if we remove hydrogens
          for (let i = 0; i < fragment.getAllAtoms(); i++) {
            fragment.setAtomMapNo(i, mappings.indexOf(i) + 1);
          }
          fragment.removeExplicitHydrogens();
          makeRacemic(fragment);
          // we encode atom characteristics in the query features
          addQueryFeaturesAndRemoveMapNo(fragment, molecule);
          fragments.push(fragment);
        }
      }
      return fragments;
    }
    /**
     * If the atom is not an halogen, X or an hydrogen
     * we add query features to the atom
     * This includes aromaticity, ring size, number of hydrogens
     * @param fragment
     * @param molecule
     */
    function addQueryFeaturesAndRemoveMapNo(fragment, molecule) {
      const Molecule = molecule.getOCL().Molecule;
      for (let i = 0; i < fragment.getAllAtoms(); i++) {
        const mapping = fragment.getAtomMapNo(i) - 1;
        fragment.setAtomMapNo(i, 0);
        if ([1, 9, 17, 35, 53, getXAtomicNumber(molecule)].includes(fragment.getAtomicNo(i))) {
          continue;
        }
        // aromaticity
        const isAromatic = molecule.isAromaticAtom(mapping);
        if (isAromatic) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFAromatic, true);
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNotAromatic, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFAromatic, false);
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNotAromatic, true);
        }
        // cycles
        const smallestRing = molecule.getAtomRingSize(mapping);
        switch (smallestRing) {
          case 0:
            break;
          case 3:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize3, true);
            break;
          case 4:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize4, true);
            break;
          case 5:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize5, true);
            break;
          case 6:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize6, true);
            break;
          case 7:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSize7, true);
            break;
          default:
            fragment.setAtomQueryFeature(i, Molecule.cAtomQFRingSizeLarge, true);
        }
        const nbHydrogens = molecule.getAllHydrogens(mapping);
        if (nbHydrogens === 0) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot0Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot0Hydrogen, true);
        }
        if (nbHydrogens === 1) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot1Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot1Hydrogen, true);
        }
        if (nbHydrogens === 2) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot2Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot2Hydrogen, true);
        }
        if (nbHydrogens === 3) {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot3Hydrogen, false);
        } else {
          fragment.setAtomQueryFeature(i, Molecule.cAtomQFNot3Hydrogen, true);
        }
      }
    }
    // tagging atoms may change the order of the atoms because hydrogens must be at the end of the file
    // in order to remember the rootAtoms we will tag before
    function internalTagAtoms(molecule, tagAtoms, rootAtoms, tagAtomFct) {
      const OCL = molecule.getOCL();
      if (tagAtoms) {
        for (let i = 0; i < molecule.getAllAtoms(); i++) {
          molecule.setAtomMapNo(i, i + 1);
        }
        if (tagAtoms.length > 0) {
          for (const atom of tagAtoms) {
            tagAtomFct(molecule, atom);
          }
        }
      }
      // this force reordering of atoms in order to have hydrogens at the end
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      if (rootAtoms.length > 0) {
        const mapping = new Int32Array(molecule.getAllAtoms());
        for (let i = 0; i < molecule.getAllAtoms(); i++) {
          mapping[molecule.getAtomMapNo(i) - 1] = i;
        }
        for (let i = 0; i < rootAtoms.length; i++) {
          rootAtoms[i] = mapping[rootAtoms[i]];
        }
      }
    }

    /**
     * We need to create an array of atoms
     * that contains an array of pathLength
     * that contains an array of object
     * @param molecule
     * @param options
     * @returns
     */
    function getAllAtomsPaths(molecule, options = {}) {
      const {
        maxPathLength = 5
      } = options;
      const allAtomsPaths = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        const oneAtomPaths = [];
        allAtomsPaths.push(oneAtomPaths);
        let atomPaths = [];
        atomPaths.push({
          path: [i],
          pathLength: 0
        });
        oneAtomPaths.push(atomPaths);
        let nextIndexes = [0];
        let nextAtoms = [i];
        for (let sphere = 1; sphere <= maxPathLength; sphere++) {
          atomPaths = [];
          oneAtomPaths.push(atomPaths);
          const currentIndexes = nextIndexes;
          const currentAtoms = nextAtoms;
          nextIndexes = [];
          nextAtoms = [];
          for (let i = 0; i < currentIndexes.length; i++) {
            const atom = currentAtoms[i];
            const index = currentIndexes[i];
            const previousPath = oneAtomPaths[sphere - 1]?.[index]?.path;
            if (!previousPath) {
              throw new Error(`Unexpected missing previousPath for sphere ${sphere - 1} and index ${index}`);
            }
            for (let conn = 0; conn < molecule.getAllConnAtoms(atom); conn++) {
              const connectedAtom = molecule.getConnAtom(atom, conn);
              if (previousPath.includes(connectedAtom)) continue;
              nextIndexes.push(atomPaths.length);
              nextAtoms.push(connectedAtom);
              atomPaths.push({
                path: [...previousPath, connectedAtom],
                pathLength: sphere
              });
            }
          }
        }
      }
      return allAtomsPaths;
    }

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    function getAugmentedNamespace(n) {
      if (Object.prototype.hasOwnProperty.call(n, '__esModule')) return n;
      var f = n.default;
    	if (typeof f == "function") {
    		var a = function a () {
    			var isInstance = false;
          try {
            isInstance = this instanceof a;
          } catch {}
    			if (isInstance) {
            return Reflect.construct(f, arguments, this.constructor);
    			}
    			return f.apply(this, arguments);
    		};
    		a.prototype = f.prototype;
      } else a = {};
      Object.defineProperty(a, '__esModule', {value: true});
    	Object.keys(n).forEach(function (k) {
    		var d = Object.getOwnPropertyDescriptor(n, k);
    		Object.defineProperty(a, k, d.get ? d : {
    			enumerable: true,
    			get: function () {
    				return n[k];
    			}
    		});
    	});
    	return a;
    }

    var matrix = {};

    // eslint-disable-next-line @typescript-eslint/unbound-method
    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
     *
     * @param value - Object to check.
     * @returns True if the object is an array or a typed array.
     */
    function isAnyArray$1(value) {
      const tag = toString.call(value);
      return tag.endsWith('Array]') && !tag.includes('Big');
    }

    var libEsm = /*#__PURE__*/Object.freeze({
        __proto__: null,
        isAnyArray: isAnyArray$1
    });

    var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

    function max(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var maxValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] > maxValue) maxValue = input[i];
      }
      return maxValue;
    }

    function min(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var minValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] < minValue) minValue = input[i];
      }
      return minValue;
    }

    function rescale$1(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      } else if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var output;
      if (options.output !== undefined) {
        if (!isAnyArray$1(options.output)) {
          throw new TypeError('output option must be an array if specified');
        }
        output = options.output;
      } else {
        output = new Array(input.length);
      }
      var currentMin = min(input);
      var currentMax = max(input);
      if (currentMin === currentMax) {
        throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
      }
      var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
      if (minValue >= maxValue) {
        throw new RangeError('min option must be smaller than max option');
      }
      var factor = (maxValue - minValue) / (currentMax - currentMin);
      for (var i = 0; i < input.length; i++) {
        output[i] = (input[i] - currentMin) * factor + minValue;
      }
      return output;
    }

    var libEs6 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        default: rescale$1
    });

    var require$$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

    Object.defineProperty(matrix, '__esModule', {
      value: true
    });
    var isAnyArray = require$$0;
    var rescale = require$$1;
    const indent = ' '.repeat(2);
    const indentData = ' '.repeat(4);

    /**
     * @this {Matrix}
     * @returns {string}
     */
    function inspectMatrix() {
      return inspectMatrixWithOptions(this);
    }
    function inspectMatrixWithOptions(matrix, options = {}) {
      const {
        maxRows = 15,
        maxColumns = 10,
        maxNumSize = 8,
        padMinus = 'auto'
      } = options;
      return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
    }
    function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
      const {
        rows,
        columns
      } = matrix;
      const maxI = Math.min(rows, maxRows);
      const maxJ = Math.min(columns, maxColumns);
      const result = [];
      if (padMinus === 'auto') {
        padMinus = false;
        loop: for (let i = 0; i < maxI; i++) {
          for (let j = 0; j < maxJ; j++) {
            if (matrix.get(i, j) < 0) {
              padMinus = true;
              break loop;
            }
          }
        }
      }
      for (let i = 0; i < maxI; i++) {
        let line = [];
        for (let j = 0; j < maxJ; j++) {
          line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
        }
        result.push(`${line.join(' ')}`);
      }
      if (maxJ !== columns) {
        result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
      }
      if (maxI !== rows) {
        result.push(`... ${rows - maxRows} more rows`);
      }
      return result.join(`\n${indentData}`);
    }
    function formatNumber(num, maxNumSize, padMinus) {
      return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
    }
    function formatNumber2(num, len) {
      // small.length numbers should be as is
      let str = num.toString();
      if (str.length <= len) return str;

      // (7)'0.00123' is better then (7)'1.23e-2'
      // (8)'0.000123' is worse then (7)'1.23e-3',
      let fix = num.toFixed(len);
      if (fix.length > len) {
        fix = num.toFixed(Math.max(0, len - (fix.length - len)));
      }
      if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
        return fix;
      }

      // well, if it's still too long the user should've used longer numbers
      let exp = num.toExponential(len);
      if (exp.length > len) {
        exp = num.toExponential(Math.max(0, len - (exp.length - len)));
      }
      return exp.slice(0);
    }
    function installMathOperations(AbstractMatrix, Matrix) {
      AbstractMatrix.prototype.add = function add(value) {
        if (typeof value === 'number') return this.addS(value);
        return this.addM(value);
      };
      AbstractMatrix.prototype.addS = function addS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.addM = function addM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.add = function add(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.add(value);
      };
      AbstractMatrix.prototype.sub = function sub(value) {
        if (typeof value === 'number') return this.subS(value);
        return this.subM(value);
      };
      AbstractMatrix.prototype.subS = function subS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.subM = function subM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.sub = function sub(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sub(value);
      };
      AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
      AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
      AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
      AbstractMatrix.subtract = AbstractMatrix.sub;
      AbstractMatrix.prototype.mul = function mul(value) {
        if (typeof value === 'number') return this.mulS(value);
        return this.mulM(value);
      };
      AbstractMatrix.prototype.mulS = function mulS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.mulM = function mulM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mul = function mul(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mul(value);
      };
      AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
      AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
      AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
      AbstractMatrix.multiply = AbstractMatrix.mul;
      AbstractMatrix.prototype.div = function div(value) {
        if (typeof value === 'number') return this.divS(value);
        return this.divM(value);
      };
      AbstractMatrix.prototype.divS = function divS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.divM = function divM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.div = function div(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.div(value);
      };
      AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
      AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
      AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
      AbstractMatrix.divide = AbstractMatrix.div;
      AbstractMatrix.prototype.mod = function mod(value) {
        if (typeof value === 'number') return this.modS(value);
        return this.modM(value);
      };
      AbstractMatrix.prototype.modS = function modS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.modM = function modM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mod = function mod(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mod(value);
      };
      AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
      AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
      AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
      AbstractMatrix.modulus = AbstractMatrix.mod;
      AbstractMatrix.prototype.and = function and(value) {
        if (typeof value === 'number') return this.andS(value);
        return this.andM(value);
      };
      AbstractMatrix.prototype.andS = function andS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.andM = function andM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.and = function and(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.and(value);
      };
      AbstractMatrix.prototype.or = function or(value) {
        if (typeof value === 'number') return this.orS(value);
        return this.orM(value);
      };
      AbstractMatrix.prototype.orS = function orS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.orM = function orM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.or = function or(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.or(value);
      };
      AbstractMatrix.prototype.xor = function xor(value) {
        if (typeof value === 'number') return this.xorS(value);
        return this.xorM(value);
      };
      AbstractMatrix.prototype.xorS = function xorS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.xorM = function xorM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.xor = function xor(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.xor(value);
      };
      AbstractMatrix.prototype.leftShift = function leftShift(value) {
        if (typeof value === 'number') return this.leftShiftS(value);
        return this.leftShiftM(value);
      };
      AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.leftShift = function leftShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.leftShift(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
        if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
        return this.signPropagatingRightShiftM(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.signPropagatingRightShift(value);
      };
      AbstractMatrix.prototype.rightShift = function rightShift(value) {
        if (typeof value === 'number') return this.rightShiftS(value);
        return this.rightShiftM(value);
      };
      AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.rightShift = function rightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.rightShift(value);
      };
      AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
      AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
      AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
      AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
      AbstractMatrix.prototype.not = function not() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, ~this.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.not = function not(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.not();
      };
      AbstractMatrix.prototype.abs = function abs() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.abs(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.abs = function abs(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.abs();
      };
      AbstractMatrix.prototype.acos = function acos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acos = function acos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acos();
      };
      AbstractMatrix.prototype.acosh = function acosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acosh = function acosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acosh();
      };
      AbstractMatrix.prototype.asin = function asin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asin = function asin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asin();
      };
      AbstractMatrix.prototype.asinh = function asinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asinh = function asinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asinh();
      };
      AbstractMatrix.prototype.atan = function atan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atan = function atan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atan();
      };
      AbstractMatrix.prototype.atanh = function atanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atanh = function atanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atanh();
      };
      AbstractMatrix.prototype.cbrt = function cbrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cbrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cbrt = function cbrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cbrt();
      };
      AbstractMatrix.prototype.ceil = function ceil() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.ceil(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.ceil = function ceil(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.ceil();
      };
      AbstractMatrix.prototype.clz32 = function clz32() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.clz32(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.clz32 = function clz32(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.clz32();
      };
      AbstractMatrix.prototype.cos = function cos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cos = function cos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cos();
      };
      AbstractMatrix.prototype.cosh = function cosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cosh = function cosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cosh();
      };
      AbstractMatrix.prototype.exp = function exp() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.exp(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.exp = function exp(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.exp();
      };
      AbstractMatrix.prototype.expm1 = function expm1() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.expm1(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.expm1 = function expm1(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.expm1();
      };
      AbstractMatrix.prototype.floor = function floor() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.floor(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.floor = function floor(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.floor();
      };
      AbstractMatrix.prototype.fround = function fround() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.fround(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.fround = function fround(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.fround();
      };
      AbstractMatrix.prototype.log = function log() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log = function log(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log();
      };
      AbstractMatrix.prototype.log1p = function log1p() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log1p(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log1p = function log1p(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log1p();
      };
      AbstractMatrix.prototype.log10 = function log10() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log10(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log10 = function log10(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log10();
      };
      AbstractMatrix.prototype.log2 = function log2() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log2(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log2 = function log2(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log2();
      };
      AbstractMatrix.prototype.round = function round() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.round(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.round = function round(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.round();
      };
      AbstractMatrix.prototype.sign = function sign() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sign(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sign = function sign(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sign();
      };
      AbstractMatrix.prototype.sin = function sin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sin = function sin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sin();
      };
      AbstractMatrix.prototype.sinh = function sinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sinh = function sinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sinh();
      };
      AbstractMatrix.prototype.sqrt = function sqrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sqrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sqrt = function sqrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sqrt();
      };
      AbstractMatrix.prototype.tan = function tan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tan = function tan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tan();
      };
      AbstractMatrix.prototype.tanh = function tanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tanh = function tanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tanh();
      };
      AbstractMatrix.prototype.trunc = function trunc() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.trunc(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.trunc = function trunc(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.trunc();
      };
      AbstractMatrix.pow = function pow(matrix, arg0) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.pow(arg0);
      };
      AbstractMatrix.prototype.pow = function pow(value) {
        if (typeof value === 'number') return this.powS(value);
        return this.powM(value);
      };
      AbstractMatrix.prototype.powS = function powS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.powM = function powM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ** matrix.get(i, j));
          }
        }
        return this;
      };
    }

    /**
     * @private
     * Check that a row index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkRowIndex(matrix, index, outer) {
      let max = outer ? matrix.rows : matrix.rows - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Row index out of range');
      }
    }

    /**
     * @private
     * Check that a column index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkColumnIndex(matrix, index, outer) {
      let max = outer ? matrix.columns : matrix.columns - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Column index out of range');
      }
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkRowVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.columns) {
        throw new RangeError('vector size must be the same as the number of columns');
      }
      return vector;
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkColumnVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.rows) {
        throw new RangeError('vector size must be the same as the number of rows');
      }
      return vector;
    }
    function checkRowIndices(matrix, rowIndices) {
      if (!isAnyArray.isAnyArray(rowIndices)) {
        throw new TypeError('row indices must be an array');
      }
      for (let i = 0; i < rowIndices.length; i++) {
        if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
          throw new RangeError('row indices are out of range');
        }
      }
    }
    function checkColumnIndices(matrix, columnIndices) {
      if (!isAnyArray.isAnyArray(columnIndices)) {
        throw new TypeError('column indices must be an array');
      }
      for (let i = 0; i < columnIndices.length; i++) {
        if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
          throw new RangeError('column indices are out of range');
        }
      }
    }
    function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
      if (arguments.length !== 5) {
        throw new RangeError('expected 4 arguments');
      }
      checkNumber('startRow', startRow);
      checkNumber('endRow', endRow);
      checkNumber('startColumn', startColumn);
      checkNumber('endColumn', endColumn);
      if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
        throw new RangeError('Submatrix indices are out of range');
      }
    }
    function newArray(length, value = 0) {
      let array = [];
      for (let i = 0; i < length; i++) {
        array.push(value);
      }
      return array;
    }
    function checkNumber(name, value) {
      if (typeof value !== 'number') {
        throw new TypeError(`${name} must be a number`);
      }
    }
    function checkNonEmpty(matrix) {
      if (matrix.isEmpty()) {
        throw new Error('Empty matrix has no elements to index');
      }
    }
    function sumByRow(matrix) {
      let sum = newArray(matrix.rows);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumByColumn(matrix) {
      let sum = newArray(matrix.columns);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumAll(matrix) {
      let v = 0;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v += matrix.get(i, j);
        }
      }
      return v;
    }
    function productByRow(matrix) {
      let sum = newArray(matrix.rows, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productByColumn(matrix) {
      let sum = newArray(matrix.columns, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productAll(matrix) {
      let v = 1;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v *= matrix.get(i, j);
        }
      }
      return v;
    }
    function varianceByRow(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let i = 0; i < rows; i++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean[i];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / cols) / cols);
        }
      }
      return variance;
    }
    function varianceByColumn(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let j = 0; j < cols; j++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let i = 0; i < rows; i++) {
          x = matrix.get(i, j) - mean[j];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / rows) / rows);
        }
      }
      return variance;
    }
    function varianceAll(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const size = rows * cols;
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean;
          sum1 += x;
          sum2 += x * x;
        }
      }
      if (unbiased) {
        return (sum2 - sum1 * sum1 / size) / (size - 1);
      } else {
        return (sum2 - sum1 * sum1 / size) / size;
      }
    }
    function centerByRow(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[i]);
        }
      }
    }
    function centerByColumn(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[j]);
        }
      }
    }
    function centerAll(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean);
        }
      }
    }
    function getScaleByRow(matrix) {
      const scale = [];
      for (let i = 0; i < matrix.rows; i++) {
        let sum = 0;
        for (let j = 0; j < matrix.columns; j++) {
          sum += matrix.get(i, j) ** 2 / (matrix.columns - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByRow(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[i]);
        }
      }
    }
    function getScaleByColumn(matrix) {
      const scale = [];
      for (let j = 0; j < matrix.columns; j++) {
        let sum = 0;
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / (matrix.rows - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByColumn(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[j]);
        }
      }
    }
    function getScaleAll(matrix) {
      const divider = matrix.size - 1;
      let sum = 0;
      for (let j = 0; j < matrix.columns; j++) {
        for (let i = 0; i < matrix.rows; i++) {
          sum += matrix.get(i, j) ** 2 / divider;
        }
      }
      return Math.sqrt(sum);
    }
    function scaleAll(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale);
        }
      }
    }
    class AbstractMatrix {
      static from1DArray(newRows, newColumns, newData) {
        let length = newRows * newColumns;
        if (length !== newData.length) {
          throw new RangeError('data length does not match given dimensions');
        }
        let newMatrix = new Matrix$1(newRows, newColumns);
        for (let row = 0; row < newRows; row++) {
          for (let column = 0; column < newColumns; column++) {
            newMatrix.set(row, column, newData[row * newColumns + column]);
          }
        }
        return newMatrix;
      }
      static rowVector(newData) {
        let vector = new Matrix$1(1, newData.length);
        for (let i = 0; i < newData.length; i++) {
          vector.set(0, i, newData[i]);
        }
        return vector;
      }
      static columnVector(newData) {
        let vector = new Matrix$1(newData.length, 1);
        for (let i = 0; i < newData.length; i++) {
          vector.set(i, 0, newData[i]);
        }
        return vector;
      }
      static zeros(rows, columns) {
        return new Matrix$1(rows, columns);
      }
      static ones(rows, columns) {
        return new Matrix$1(rows, columns).fill(1);
      }
      static rand(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          random = Math.random
        } = options;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.set(i, j, random());
          }
        }
        return matrix;
      }
      static randInt(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1000,
          random = Math.random
        } = options;
        if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
        if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let interval = max - min;
        let matrix = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            let value = min + Math.round(random() * interval);
            matrix.set(i, j, value);
          }
        }
        return matrix;
      }
      static eye(rows, columns, value) {
        if (columns === undefined) columns = rows;
        if (value === undefined) value = 1;
        let min = Math.min(rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, value);
        }
        return matrix;
      }
      static diag(data, rows, columns) {
        let l = data.length;
        if (rows === undefined) rows = l;
        if (columns === undefined) columns = rows;
        let min = Math.min(l, rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, data[i]);
        }
        return matrix;
      }
      static min(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static max(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new this(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static checkMatrix(value) {
        return AbstractMatrix.isMatrix(value) ? value : new Matrix$1(value);
      }
      static isMatrix(value) {
        return value != null && value.klass === 'Matrix';
      }
      get size() {
        return this.rows * this.columns;
      }
      apply(callback) {
        if (typeof callback !== 'function') {
          throw new TypeError('callback must be a function');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            callback.call(this, i, j);
          }
        }
        return this;
      }
      to1DArray() {
        let array = [];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            array.push(this.get(i, j));
          }
        }
        return array;
      }
      to2DArray() {
        let copy = [];
        for (let i = 0; i < this.rows; i++) {
          copy.push([]);
          for (let j = 0; j < this.columns; j++) {
            copy[i].push(this.get(i, j));
          }
        }
        return copy;
      }
      toJSON() {
        return this.to2DArray();
      }
      isRowVector() {
        return this.rows === 1;
      }
      isColumnVector() {
        return this.columns === 1;
      }
      isVector() {
        return this.rows === 1 || this.columns === 1;
      }
      isSquare() {
        return this.rows === this.columns;
      }
      isEmpty() {
        return this.rows === 0 || this.columns === 0;
      }
      isSymmetric() {
        if (this.isSquare()) {
          for (let i = 0; i < this.rows; i++) {
            for (let j = 0; j <= i; j++) {
              if (this.get(i, j) !== this.get(j, i)) {
                return false;
              }
            }
          }
          return true;
        }
        return false;
      }
      isDistance() {
        if (!this.isSymmetric()) return false;
        for (let i = 0; i < this.rows; i++) {
          if (this.get(i, i) !== 0) return false;
        }
        return true;
      }
      isEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isEchelonForm = true;
        let checked = false;
        while (i < this.rows && isEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isEchelonForm = false;
              checked = true;
            }
          }
          i++;
        }
        return isEchelonForm;
      }
      isReducedEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isReducedEchelonForm = true;
        let checked = false;
        while (i < this.rows && isReducedEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isReducedEchelonForm = false;
              checked = true;
            }
          }
          for (let k = j + 1; k < this.rows; k++) {
            if (this.get(i, k) !== 0) {
              isReducedEchelonForm = false;
            }
          }
          i++;
        }
        return isReducedEchelonForm;
      }
      echelonForm() {
        let result = this.clone();
        let h = 0;
        let k = 0;
        while (h < result.rows && k < result.columns) {
          let iMax = h;
          for (let i = h; i < result.rows; i++) {
            if (result.get(i, k) > result.get(iMax, k)) {
              iMax = i;
            }
          }
          if (result.get(iMax, k) === 0) {
            k++;
          } else {
            result.swapRows(h, iMax);
            let tmp = result.get(h, k);
            for (let j = k; j < result.columns; j++) {
              result.set(h, j, result.get(h, j) / tmp);
            }
            for (let i = h + 1; i < result.rows; i++) {
              let factor = result.get(i, k) / result.get(h, k);
              result.set(i, k, 0);
              for (let j = k + 1; j < result.columns; j++) {
                result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
              }
            }
            h++;
            k++;
          }
        }
        return result;
      }
      reducedEchelonForm() {
        let result = this.echelonForm();
        let m = result.columns;
        let n = result.rows;
        let h = n - 1;
        while (h >= 0) {
          if (result.maxRow(h) === 0) {
            h--;
          } else {
            let p = 0;
            let pivot = false;
            while (p < n && pivot === false) {
              if (result.get(h, p) === 1) {
                pivot = true;
              } else {
                p++;
              }
            }
            for (let i = 0; i < h; i++) {
              let factor = result.get(i, p);
              for (let j = p; j < m; j++) {
                let tmp = result.get(i, j) - factor * result.get(h, j);
                result.set(i, j, tmp);
              }
            }
            h--;
          }
        }
        return result;
      }
      set() {
        throw new Error('set method is unimplemented');
      }
      get() {
        throw new Error('get method is unimplemented');
      }
      repeat(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          rows = 1,
          columns = 1
        } = options;
        if (!Number.isInteger(rows) || rows <= 0) {
          throw new TypeError('rows must be a positive integer');
        }
        if (!Number.isInteger(columns) || columns <= 0) {
          throw new TypeError('columns must be a positive integer');
        }
        let matrix = new Matrix$1(this.rows * rows, this.columns * columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.setSubMatrix(this, this.rows * i, this.columns * j);
          }
        }
        return matrix;
      }
      fill(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, value);
          }
        }
        return this;
      }
      neg() {
        return this.mulS(-1);
      }
      getRow(index) {
        checkRowIndex(this, index);
        let row = [];
        for (let i = 0; i < this.columns; i++) {
          row.push(this.get(index, i));
        }
        return row;
      }
      getRowVector(index) {
        return Matrix$1.rowVector(this.getRow(index));
      }
      setRow(index, array) {
        checkRowIndex(this, index);
        array = checkRowVector(this, array);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, array[i]);
        }
        return this;
      }
      swapRows(row1, row2) {
        checkRowIndex(this, row1);
        checkRowIndex(this, row2);
        for (let i = 0; i < this.columns; i++) {
          let temp = this.get(row1, i);
          this.set(row1, i, this.get(row2, i));
          this.set(row2, i, temp);
        }
        return this;
      }
      getColumn(index) {
        checkColumnIndex(this, index);
        let column = [];
        for (let i = 0; i < this.rows; i++) {
          column.push(this.get(i, index));
        }
        return column;
      }
      getColumnVector(index) {
        return Matrix$1.columnVector(this.getColumn(index));
      }
      setColumn(index, array) {
        checkColumnIndex(this, index);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, array[i]);
        }
        return this;
      }
      swapColumns(column1, column2) {
        checkColumnIndex(this, column1);
        checkColumnIndex(this, column2);
        for (let i = 0; i < this.rows; i++) {
          let temp = this.get(i, column1);
          this.set(i, column1, this.get(i, column2));
          this.set(i, column2, temp);
        }
        return this;
      }
      addRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[j]);
          }
        }
        return this;
      }
      subRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[j]);
          }
        }
        return this;
      }
      mulRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[j]);
          }
        }
        return this;
      }
      divRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[j]);
          }
        }
        return this;
      }
      addColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[i]);
          }
        }
        return this;
      }
      subColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[i]);
          }
        }
        return this;
      }
      mulColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[i]);
          }
        }
        return this;
      }
      divColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[i]);
          }
        }
        return this;
      }
      mulRow(index, value) {
        checkRowIndex(this, index);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, this.get(index, i) * value);
        }
        return this;
      }
      mulColumn(index, value) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, this.get(i, index) * value);
        }
        return this;
      }
      max(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[row]) {
                    max[row] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case 'column':
            {
              const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[column]) {
                    max[column] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case undefined:
            {
              let max = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max) {
                    max = this.get(row, column);
                  }
                }
              }
              return max;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      maxIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) > v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      min(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[row]) {
                    min[row] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case 'column':
            {
              const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[column]) {
                    min[column] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case undefined:
            {
              let min = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min) {
                    min = this.get(row, column);
                  }
                }
              }
              return min;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      minIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) < v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      maxRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      maxRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      minRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      minRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      maxColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      maxColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      minColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      minColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      diag() {
        let min = Math.min(this.rows, this.columns);
        let diag = [];
        for (let i = 0; i < min; i++) {
          diag.push(this.get(i, i));
        }
        return diag;
      }
      norm(type = 'frobenius') {
        switch (type) {
          case 'max':
            return this.max();
          case 'frobenius':
            return Math.sqrt(this.dot(this));
          default:
            throw new RangeError(`unknown norm type: ${type}`);
        }
      }
      cumulativeSum() {
        let sum = 0;
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            sum += this.get(i, j);
            this.set(i, j, sum);
          }
        }
        return this;
      }
      dot(vector2) {
        if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
        let vector1 = this.to1DArray();
        if (vector1.length !== vector2.length) {
          throw new RangeError('vectors do not have the same size');
        }
        let dot = 0;
        for (let i = 0; i < vector1.length; i++) {
          dot += vector1[i] * vector2[i];
        }
        return dot;
      }
      mmul(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.columns;
        let result = new Matrix$1(m, p);
        let Bcolj = new Float64Array(n);
        for (let j = 0; j < p; j++) {
          for (let k = 0; k < n; k++) {
            Bcolj[k] = other.get(k, j);
          }
          for (let i = 0; i < m; i++) {
            let s = 0;
            for (let k = 0; k < n; k++) {
              s += this.get(i, k) * Bcolj[k];
            }
            result.set(i, j, s);
          }
        }
        return result;
      }
      mpow(scalar) {
        if (!this.isSquare()) {
          throw new RangeError('Matrix must be square');
        }
        if (!Number.isInteger(scalar) || scalar < 0) {
          throw new RangeError('Exponent must be a non-negative integer');
        }
        // Russian Peasant exponentiation, i.e. exponentiation by squaring
        let result = Matrix$1.eye(this.rows);
        let bb = this;
        // Note: Don't bit shift. In JS, that would truncate at 32 bits
        for (let e = scalar; e >= 1; e /= 2) {
          if ((e & 1) !== 0) {
            result = result.mmul(bb);
          }
          bb = bb.mmul(bb);
        }
        return result;
      }
      strassen2x2(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(2, 2);
        const a11 = this.get(0, 0);
        const b11 = other.get(0, 0);
        const a12 = this.get(0, 1);
        const b12 = other.get(0, 1);
        const a21 = this.get(1, 0);
        const b21 = other.get(1, 0);
        const a22 = this.get(1, 1);
        const b22 = other.get(1, 1);

        // Compute intermediate values.
        const m1 = (a11 + a22) * (b11 + b22);
        const m2 = (a21 + a22) * b11;
        const m3 = a11 * (b12 - b22);
        const m4 = a22 * (b21 - b11);
        const m5 = (a11 + a12) * b22;
        const m6 = (a21 - a11) * (b11 + b12);
        const m7 = (a12 - a22) * (b21 + b22);

        // Combine intermediate values into the output.
        const c00 = m1 + m4 - m5 + m7;
        const c01 = m3 + m5;
        const c10 = m2 + m4;
        const c11 = m1 - m2 + m3 + m6;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        return result;
      }
      strassen3x3(other) {
        other = Matrix$1.checkMatrix(other);
        let result = new Matrix$1(3, 3);
        const a00 = this.get(0, 0);
        const a01 = this.get(0, 1);
        const a02 = this.get(0, 2);
        const a10 = this.get(1, 0);
        const a11 = this.get(1, 1);
        const a12 = this.get(1, 2);
        const a20 = this.get(2, 0);
        const a21 = this.get(2, 1);
        const a22 = this.get(2, 2);
        const b00 = other.get(0, 0);
        const b01 = other.get(0, 1);
        const b02 = other.get(0, 2);
        const b10 = other.get(1, 0);
        const b11 = other.get(1, 1);
        const b12 = other.get(1, 2);
        const b20 = other.get(2, 0);
        const b21 = other.get(2, 1);
        const b22 = other.get(2, 2);
        const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
        const m2 = (a00 - a10) * (-b01 + b11);
        const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
        const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
        const m5 = (a10 + a11) * (-b00 + b01);
        const m6 = a00 * b00;
        const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
        const m8 = (-a00 + a20) * (b02 - b12);
        const m9 = (a20 + a21) * (-b00 + b02);
        const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
        const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
        const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
        const m13 = (a02 - a22) * (b11 - b21);
        const m14 = a02 * b20;
        const m15 = (a21 + a22) * (-b20 + b21);
        const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
        const m17 = (a02 - a12) * (b12 - b22);
        const m18 = (a11 + a12) * (-b20 + b22);
        const m19 = a01 * b10;
        const m20 = a12 * b21;
        const m21 = a10 * b02;
        const m22 = a20 * b01;
        const m23 = a22 * b22;
        const c00 = m6 + m14 + m19;
        const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
        const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
        const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
        const c11 = m2 + m4 + m5 + m6 + m20;
        const c12 = m14 + m16 + m17 + m18 + m21;
        const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
        const c21 = m12 + m13 + m14 + m15 + m22;
        const c22 = m6 + m7 + m8 + m9 + m23;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(0, 2, c02);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        result.set(1, 2, c12);
        result.set(2, 0, c20);
        result.set(2, 1, c21);
        result.set(2, 2, c22);
        return result;
      }
      mmulStrassen(y) {
        y = Matrix$1.checkMatrix(y);
        let x = this.clone();
        let r1 = x.rows;
        let c1 = x.columns;
        let r2 = y.rows;
        let c2 = y.columns;
        if (c1 !== r2) {
          // eslint-disable-next-line no-console
          console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
        }

        // Put a matrix into the top left of a matrix of zeros.
        // `rows` and `cols` are the dimensions of the output matrix.
        function embed(mat, rows, cols) {
          let r = mat.rows;
          let c = mat.columns;
          if (r === rows && c === cols) {
            return mat;
          } else {
            let resultat = AbstractMatrix.zeros(rows, cols);
            resultat = resultat.setSubMatrix(mat, 0, 0);
            return resultat;
          }
        }

        // Make sure both matrices are the same size.
        // This is exclusively for simplicity:
        // this algorithm can be implemented with matrices of different sizes.

        let r = Math.max(r1, r2);
        let c = Math.max(c1, c2);
        x = embed(x, r, c);
        y = embed(y, r, c);

        // Our recursive multiplication function.
        function blockMult(a, b, rows, cols) {
          // For small matrices, resort to naive multiplication.
          if (rows <= 512 || cols <= 512) {
            return a.mmul(b); // a is equivalent to this
          }

          // Apply dynamic padding.
          if (rows % 2 === 1 && cols % 2 === 1) {
            a = embed(a, rows + 1, cols + 1);
            b = embed(b, rows + 1, cols + 1);
          } else if (rows % 2 === 1) {
            a = embed(a, rows + 1, cols);
            b = embed(b, rows + 1, cols);
          } else if (cols % 2 === 1) {
            a = embed(a, rows, cols + 1);
            b = embed(b, rows, cols + 1);
          }
          let halfRows = parseInt(a.rows / 2, 10);
          let halfCols = parseInt(a.columns / 2, 10);
          // Subdivide input matrices.
          let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
          let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
          let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
          let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
          let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
          let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

          // Compute intermediate values.
          let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
          let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
          let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
          let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
          let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
          let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
          let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

          // Combine intermediate values into the output.
          let c11 = AbstractMatrix.add(m1, m4);
          c11.sub(m5);
          c11.add(m7);
          let c12 = AbstractMatrix.add(m3, m5);
          let c21 = AbstractMatrix.add(m2, m4);
          let c22 = AbstractMatrix.sub(m1, m2);
          c22.add(m3);
          c22.add(m6);

          // Crop output to the desired size (undo dynamic padding).
          let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
          result = result.setSubMatrix(c11, 0, 0);
          result = result.setSubMatrix(c12, c11.rows, 0);
          result = result.setSubMatrix(c21, 0, c11.columns);
          result = result.setSubMatrix(c22, c11.rows, c11.columns);
          return result.subMatrix(0, rows - 1, 0, cols - 1);
        }
        return blockMult(x, y, r, c);
      }
      scaleRows(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.rows; i++) {
          const row = this.getRow(i);
          if (row.length > 0) {
            rescale(row, {
              min,
              max,
              output: row
            });
          }
          newMatrix.setRow(i, row);
        }
        return newMatrix;
      }
      scaleColumns(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$1(this.rows, this.columns);
        for (let i = 0; i < this.columns; i++) {
          const column = this.getColumn(i);
          if (column.length) {
            rescale(column, {
              min,
              max,
              output: column
            });
          }
          newMatrix.setColumn(i, column);
        }
        return newMatrix;
      }
      flipRows() {
        const middle = Math.ceil(this.columns / 2);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < middle; j++) {
            let first = this.get(i, j);
            let last = this.get(i, this.columns - 1 - j);
            this.set(i, j, last);
            this.set(i, this.columns - 1 - j, first);
          }
        }
        return this;
      }
      flipColumns() {
        const middle = Math.ceil(this.rows / 2);
        for (let j = 0; j < this.columns; j++) {
          for (let i = 0; i < middle; i++) {
            let first = this.get(i, j);
            let last = this.get(this.rows - 1 - i, j);
            this.set(i, j, last);
            this.set(this.rows - 1 - i, j, first);
          }
        }
        return this;
      }
      kroneckerProduct(other) {
        other = Matrix$1.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.rows;
        let q = other.columns;
        let result = new Matrix$1(m * p, n * q);
        for (let i = 0; i < m; i++) {
          for (let j = 0; j < n; j++) {
            for (let k = 0; k < p; k++) {
              for (let l = 0; l < q; l++) {
                result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
              }
            }
          }
        }
        return result;
      }
      kroneckerSum(other) {
        other = Matrix$1.checkMatrix(other);
        if (!this.isSquare() || !other.isSquare()) {
          throw new Error('Kronecker Sum needs two Square Matrices');
        }
        let m = this.rows;
        let n = other.rows;
        let AxI = this.kroneckerProduct(Matrix$1.eye(n, n));
        let IxB = Matrix$1.eye(m, m).kroneckerProduct(other);
        return AxI.add(IxB);
      }
      transpose() {
        let result = new Matrix$1(this.columns, this.rows);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result.set(j, i, this.get(i, j));
          }
        }
        return result;
      }
      sortRows(compareFunction = compareNumbers) {
        for (let i = 0; i < this.rows; i++) {
          this.setRow(i, this.getRow(i).sort(compareFunction));
        }
        return this;
      }
      sortColumns(compareFunction = compareNumbers) {
        for (let i = 0; i < this.columns; i++) {
          this.setColumn(i, this.getColumn(i).sort(compareFunction));
        }
        return this;
      }
      subMatrix(startRow, endRow, startColumn, endColumn) {
        checkRange(this, startRow, endRow, startColumn, endColumn);
        let newMatrix = new Matrix$1(endRow - startRow + 1, endColumn - startColumn + 1);
        for (let i = startRow; i <= endRow; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
          }
        }
        return newMatrix;
      }
      subMatrixRow(indices, startColumn, endColumn) {
        if (startColumn === undefined) startColumn = 0;
        if (endColumn === undefined) endColumn = this.columns - 1;
        if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(indices.length, endColumn - startColumn + 1);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            if (indices[i] < 0 || indices[i] >= this.rows) {
              throw new RangeError(`Row index out of range: ${indices[i]}`);
            }
            newMatrix.set(i, j - startColumn, this.get(indices[i], j));
          }
        }
        return newMatrix;
      }
      subMatrixColumn(indices, startRow, endRow) {
        if (startRow === undefined) startRow = 0;
        if (endRow === undefined) endRow = this.rows - 1;
        if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$1(endRow - startRow + 1, indices.length);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startRow; j <= endRow; j++) {
            if (indices[i] < 0 || indices[i] >= this.columns) {
              throw new RangeError(`Column index out of range: ${indices[i]}`);
            }
            newMatrix.set(j - startRow, i, this.get(j, indices[i]));
          }
        }
        return newMatrix;
      }
      setSubMatrix(matrix, startRow, startColumn) {
        matrix = Matrix$1.checkMatrix(matrix);
        if (matrix.isEmpty()) {
          return this;
        }
        let endRow = startRow + matrix.rows - 1;
        let endColumn = startColumn + matrix.columns - 1;
        checkRange(this, startRow, endRow, startColumn, endColumn);
        for (let i = 0; i < matrix.rows; i++) {
          for (let j = 0; j < matrix.columns; j++) {
            this.set(startRow + i, startColumn + j, matrix.get(i, j));
          }
        }
        return this;
      }
      selection(rowIndices, columnIndices) {
        checkRowIndices(this, rowIndices);
        checkColumnIndices(this, columnIndices);
        let newMatrix = new Matrix$1(rowIndices.length, columnIndices.length);
        for (let i = 0; i < rowIndices.length; i++) {
          let rowIndex = rowIndices[i];
          for (let j = 0; j < columnIndices.length; j++) {
            let columnIndex = columnIndices[j];
            newMatrix.set(i, j, this.get(rowIndex, columnIndex));
          }
        }
        return newMatrix;
      }
      trace() {
        let min = Math.min(this.rows, this.columns);
        let trace = 0;
        for (let i = 0; i < min; i++) {
          trace += this.get(i, i);
        }
        return trace;
      }
      clone() {
        return this.constructor.copy(this, new Matrix$1(this.rows, this.columns));
      }

      /**
       * @template {AbstractMatrix} M
       * @param {AbstractMatrix} from
       * @param {M} to
       * @return {M}
       */
      static copy(from, to) {
        for (const [row, column, value] of from.entries()) {
          to.set(row, column, value);
        }
        return to;
      }
      sum(by) {
        switch (by) {
          case 'row':
            return sumByRow(this);
          case 'column':
            return sumByColumn(this);
          case undefined:
            return sumAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      product(by) {
        switch (by) {
          case 'row':
            return productByRow(this);
          case 'column':
            return productByColumn(this);
          case undefined:
            return productAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      mean(by) {
        const sum = this.sum(by);
        switch (by) {
          case 'row':
            {
              for (let i = 0; i < this.rows; i++) {
                sum[i] /= this.columns;
              }
              return sum;
            }
          case 'column':
            {
              for (let i = 0; i < this.columns; i++) {
                sum[i] /= this.rows;
              }
              return sum;
            }
          case undefined:
            return sum / this.size;
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      variance(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          unbiased = true,
          mean = this.mean(by)
        } = options;
        if (typeof unbiased !== 'boolean') {
          throw new TypeError('unbiased must be a boolean');
        }
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByRow(this, unbiased, mean);
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByColumn(this, unbiased, mean);
            }
          case undefined:
            {
              if (typeof mean !== 'number') {
                throw new TypeError('mean must be a number');
              }
              return varianceAll(this, unbiased, mean);
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      standardDeviation(by, options) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        const variance = this.variance(by, options);
        if (by === undefined) {
          return Math.sqrt(variance);
        } else {
          for (let i = 0; i < variance.length; i++) {
            variance[i] = Math.sqrt(variance[i]);
          }
          return variance;
        }
      }
      center(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          center = this.mean(by)
        } = options;
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByRow(this, center);
              return this;
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByColumn(this, center);
              return this;
            }
          case undefined:
            {
              if (typeof center !== 'number') {
                throw new TypeError('center must be a number');
              }
              centerAll(this, center);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      scale(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        let scale = options.scale;
        switch (by) {
          case 'row':
            {
              if (scale === undefined) {
                scale = getScaleByRow(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByRow(this, scale);
              return this;
            }
          case 'column':
            {
              if (scale === undefined) {
                scale = getScaleByColumn(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByColumn(this, scale);
              return this;
            }
          case undefined:
            {
              if (scale === undefined) {
                scale = getScaleAll(this);
              } else if (typeof scale !== 'number') {
                throw new TypeError('scale must be a number');
              }
              scaleAll(this, scale);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      toString(options) {
        return inspectMatrixWithOptions(this, options);
      }
      [Symbol.iterator]() {
        return this.entries();
      }

      /**
       * iterator from left to right, from top to bottom
       * yield [row, column, value]
       * @returns {Generator<[number, number, number], void, void>}
       */
      *entries() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield [row, col, this.get(row, col)];
          }
        }
      }

      /**
       * iterator from left to right, from top to bottom
       * yield value
       * @returns {Generator<number, void, void>}
       */
      *values() {
        for (let row = 0; row < this.rows; row++) {
          for (let col = 0; col < this.columns; col++) {
            yield this.get(row, col);
          }
        }
      }
    }
    AbstractMatrix.prototype.klass = 'Matrix';
    if (typeof Symbol !== 'undefined') {
      AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
    }
    function compareNumbers(a, b) {
      return a - b;
    }
    function isArrayOfNumbers(array) {
      return array.every(element => {
        return typeof element === 'number';
      });
    }

    // Synonyms
    AbstractMatrix.random = AbstractMatrix.rand;
    AbstractMatrix.randomInt = AbstractMatrix.randInt;
    AbstractMatrix.diagonal = AbstractMatrix.diag;
    AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
    AbstractMatrix.identity = AbstractMatrix.eye;
    AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
    AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
    let Matrix$1 = class Matrix extends AbstractMatrix {
      /**
       * @type {Float64Array[]}
       */
      data;

      /**
       * Init an empty matrix
       * @param {number} nRows
       * @param {number} nColumns
       */
      #initData(nRows, nColumns) {
        this.data = [];
        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
        this.rows = nRows;
        this.columns = nColumns;
      }
      constructor(nRows, nColumns) {
        super();
        if (Matrix.isMatrix(nRows)) {
          this.#initData(nRows.rows, nRows.columns);
          Matrix.copy(nRows, this);
        } else if (Number.isInteger(nRows) && nRows >= 0) {
          this.#initData(nRows, nColumns);
        } else if (isAnyArray.isAnyArray(nRows)) {
          // Copy the values from the 2D array
          const arrayData = nRows;
          nRows = arrayData.length;
          nColumns = nRows ? arrayData[0].length : 0;
          if (typeof nColumns !== 'number') {
            throw new TypeError('Data must be a 2D array with at least one element');
          }
          this.data = [];
          for (let i = 0; i < nRows; i++) {
            if (arrayData[i].length !== nColumns) {
              throw new RangeError('Inconsistent array dimensions');
            }
            if (!isArrayOfNumbers(arrayData[i])) {
              throw new TypeError('Input data contains non-numeric values');
            }
            this.data.push(Float64Array.from(arrayData[i]));
          }
          this.rows = nRows;
          this.columns = nColumns;
        } else {
          throw new TypeError('First argument must be a positive number or an array');
        }
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
      removeRow(index) {
        checkRowIndex(this, index);
        this.data.splice(index, 1);
        this.rows -= 1;
        return this;
      }
      addRow(index, array) {
        if (array === undefined) {
          array = index;
          index = this.rows;
        }
        checkRowIndex(this, index, true);
        array = Float64Array.from(checkRowVector(this, array));
        this.data.splice(index, 0, array);
        this.rows += 1;
        return this;
      }
      removeColumn(index) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns - 1);
          for (let j = 0; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          for (let j = index + 1; j < this.columns; j++) {
            newRow[j - 1] = this.data[i][j];
          }
          this.data[i] = newRow;
        }
        this.columns -= 1;
        return this;
      }
      addColumn(index, array) {
        if (typeof array === 'undefined') {
          array = index;
          index = this.columns;
        }
        checkColumnIndex(this, index, true);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns + 1);
          let j = 0;
          for (; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          newRow[j++] = array[i];
          for (; j < this.columns + 1; j++) {
            newRow[j] = this.data[i][j - 1];
          }
          this.data[i] = newRow;
        }
        this.columns += 1;
        return this;
      }
    };
    installMathOperations(AbstractMatrix, Matrix$1);

    /**
     * @typedef {0 | 1 | number | boolean} Mask
     */

    class SymmetricMatrix extends AbstractMatrix {
      /** @type {Matrix} */
      #matrix;
      get size() {
        return this.#matrix.size;
      }
      get rows() {
        return this.#matrix.rows;
      }
      get columns() {
        return this.#matrix.columns;
      }
      get diagonalSize() {
        return this.rows;
      }

      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isSymmetricMatrix(value) {
        return Matrix$1.isMatrix(value) && value.klassType === 'SymmetricMatrix';
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static zeros(diagonalSize) {
        return new this(diagonalSize);
      }

      /**
       * @param diagonalSize
       * @return {SymmetricMatrix}
       */
      static ones(diagonalSize) {
        return new this(diagonalSize).fill(1);
      }

      /**
       * @param {number | AbstractMatrix | ArrayLike<ArrayLike<number>>} diagonalSize
       * @return {this}
       */
      constructor(diagonalSize) {
        super();
        if (Matrix$1.isMatrix(diagonalSize)) {
          if (!diagonalSize.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
          this.#matrix = Matrix$1.copy(diagonalSize, new Matrix$1(diagonalSize.rows, diagonalSize.rows));
        } else if (Number.isInteger(diagonalSize) && diagonalSize >= 0) {
          this.#matrix = new Matrix$1(diagonalSize, diagonalSize);
        } else {
          this.#matrix = new Matrix$1(diagonalSize);
          if (!this.isSymmetric()) {
            throw new TypeError('not symmetric data');
          }
        }
      }
      clone() {
        const matrix = new SymmetricMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          matrix.set(row, col, value);
        }
        return matrix;
      }
      toMatrix() {
        return new Matrix$1(this);
      }
      get(rowIndex, columnIndex) {
        return this.#matrix.get(rowIndex, columnIndex);
      }
      set(rowIndex, columnIndex, value) {
        // symmetric set
        this.#matrix.set(rowIndex, columnIndex, value);
        this.#matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      removeCross(index) {
        // symmetric remove side
        this.#matrix.removeRow(index);
        this.#matrix.removeColumn(index);
        return this;
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }
        const row = array.slice();
        row.splice(index, 1);
        this.#matrix.addRow(index, row);
        this.#matrix.addColumn(index, array);
        return this;
      }

      /**
       * @param {Mask[]} mask
       */
      applyMask(mask) {
        if (mask.length !== this.diagonalSize) {
          throw new RangeError('Mask size do not match with matrix size');
        }

        // prepare sides to remove from matrix from mask
        /** @type {number[]} */
        const sidesToRemove = [];
        for (const [index, passthroughs] of mask.entries()) {
          if (passthroughs) continue;
          sidesToRemove.push(index);
        }
        // to remove from highest to lowest for no mutation shifting
        sidesToRemove.reverse();

        // remove sides
        for (const sideIndex of sidesToRemove) {
          this.removeCross(sideIndex);
        }
        return this;
      }

      /**
       * Compact format upper-right corner of matrix
       * iterate from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 1 2 3 4
       * B 2 5 6 7
       * C 3 6 8 9
       * D 4 7 9 10
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]`
       *
       * length is S(i=0, n=sideSize) => 10 for a 4 sideSized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;

        /** @type {number[]} */
        const compact = new Array(diagonalSize * (diagonalSize + 1) / 2);
        for (let col = 0, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       * @return {SymmetricMatrix}
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        // compactSize = (sideSize * (sideSize + 1)) / 2
        // https://mathsolver.microsoft.com/fr/solve-problem/y%20%3D%20%20x%20%60cdot%20%20%20%60frac%7B%20%20%60left(%20x%2B1%20%20%60right)%20%20%20%20%7D%7B%202%20%20%7D
        // sideSize = (Sqrt(8 × compactSize + 1) - 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) - 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a Symmetric Matrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new SymmetricMatrix(diagonalSize);
        for (let col = 0, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row;
        }
        return matrix;
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield [row, column, value]
       *
       * @returns {Generator<[number, number, number], void, void>}
       */
      *upperRightEntries() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield [row, col, value];

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }

      /**
       * half iterator upper-right-corner from left to right, from top to bottom
       * yield value
       *
       * @returns {Generator<[number, number, number], void, void>}
       */
      *upperRightValues() {
        for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
          const value = this.get(row, col);
          yield value;

          // at the end of row, move cursor to next row at diagonal position
          if (++col >= this.diagonalSize) col = ++row;
        }
      }
    }
    SymmetricMatrix.prototype.klassType = 'SymmetricMatrix';
    class DistanceMatrix extends SymmetricMatrix {
      /**
       * not the same as matrix.isSymmetric()
       * Here is to check if it's instanceof SymmetricMatrix without bundling issues
       *
       * @param value
       * @returns {boolean}
       */
      static isDistanceMatrix(value) {
        return SymmetricMatrix.isSymmetricMatrix(value) && value.klassSubType === 'DistanceMatrix';
      }
      constructor(sideSize) {
        super(sideSize);
        if (!this.isDistance()) {
          throw new TypeError('Provided arguments do no produce a distance matrix');
        }
      }
      set(rowIndex, columnIndex, value) {
        // distance matrix diagonal is 0
        if (rowIndex === columnIndex) value = 0;
        return super.set(rowIndex, columnIndex, value);
      }
      addCross(index, array) {
        if (array === undefined) {
          array = index;
          index = this.diagonalSize;
        }

        // ensure distance
        array = array.slice();
        array[index] = 0;
        return super.addCross(index, array);
      }
      toSymmetricMatrix() {
        return new SymmetricMatrix(this);
      }
      clone() {
        const matrix = new DistanceMatrix(this.diagonalSize);
        for (const [row, col, value] of this.upperRightEntries()) {
          if (row === col) continue;
          matrix.set(row, col, value);
        }
        return matrix;
      }

      /**
       * Compact format upper-right corner of matrix
       * no diagonal (only zeros)
       * iterable from left to right, from top to bottom.
       *
       * ```
       *   A B C D
       * A 0 1 2 3
       * B 1 0 4 5
       * C 2 4 0 6
       * D 3 5 6 0
       * ```
       *
       * will return compact 1D array `[1, 2, 3, 4, 5, 6]`
       *
       * length is S(i=0, n=sideSize-1) => 6 for a 4 side sized matrix
       *
       * @returns {number[]}
       */
      toCompact() {
        const {
          diagonalSize
        } = this;
        const compactLength = (diagonalSize - 1) * diagonalSize / 2;

        /** @type {number[]} */
        const compact = new Array(compactLength);
        for (let col = 1, row = 0, index = 0; index < compact.length; index++) {
          compact[index] = this.get(row, col);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return compact;
      }

      /**
       * @param {number[]} compact
       */
      static fromCompact(compact) {
        const compactSize = compact.length;
        if (compactSize === 0) {
          return new this(0);
        }

        // compactSize in Natural integer range ]0;∞]
        // compactSize = (sideSize * (sideSize - 1)) / 2
        // sideSize = (Sqrt(8 × compactSize + 1) + 1) / 2
        const diagonalSize = (Math.sqrt(8 * compactSize + 1) + 1) / 2;
        if (!Number.isInteger(diagonalSize)) {
          throw new TypeError(`This array is not a compact representation of a DistanceMatrix, ${JSON.stringify(compact)}`);
        }
        const matrix = new this(diagonalSize);
        for (let col = 1, row = 0, index = 0; index < compactSize; index++) {
          matrix.set(col, row, compact[index]);
          if (++col >= diagonalSize) col = ++row + 1;
        }
        return matrix;
      }
    }
    DistanceMatrix.prototype.klassSubType = 'DistanceMatrix';
    class BaseView extends AbstractMatrix {
      constructor(matrix, rows, columns) {
        super();
        this.matrix = matrix;
        this.rows = rows;
        this.columns = columns;
      }
    }
    class MatrixColumnView extends BaseView {
      constructor(matrix, column) {
        checkColumnIndex(matrix, column);
        super(matrix, matrix.rows, 1);
        this.column = column;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.column, value);
        return this;
      }
      get(rowIndex) {
        return this.matrix.get(rowIndex, this.column);
      }
    }
    class MatrixColumnSelectionView extends BaseView {
      constructor(matrix, columnIndices) {
        checkColumnIndices(matrix, columnIndices);
        super(matrix, matrix.rows, columnIndices.length);
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
      }
    }
    class MatrixFlipColumnView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
      }
    }
    class MatrixFlipRowView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
      }
    }
    class MatrixRowView extends BaseView {
      constructor(matrix, row) {
        checkRowIndex(matrix, row);
        super(matrix, 1, matrix.columns);
        this.row = row;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.row, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.row, columnIndex);
      }
    }
    class MatrixRowSelectionView extends BaseView {
      constructor(matrix, rowIndices) {
        checkRowIndices(matrix, rowIndices);
        super(matrix, rowIndices.length, matrix.columns);
        this.rowIndices = rowIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
      }
    }
    class MatrixSelectionView extends BaseView {
      constructor(matrix, rowIndices, columnIndices) {
        checkRowIndices(matrix, rowIndices);
        checkColumnIndices(matrix, columnIndices);
        super(matrix, rowIndices.length, columnIndices.length);
        this.rowIndices = rowIndices;
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
      }
    }
    class MatrixSubView extends BaseView {
      constructor(matrix, startRow, endRow, startColumn, endColumn) {
        checkRange(matrix, startRow, endRow, startColumn, endColumn);
        super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
        this.startRow = startRow;
        this.startColumn = startColumn;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
      }
    }
    class MatrixTransposeView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.columns, matrix.rows);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(columnIndex, rowIndex);
      }
    }
    class WrapperMatrix1D extends AbstractMatrix {
      constructor(data, options = {}) {
        const {
          rows = 1
        } = options;
        if (data.length % rows !== 0) {
          throw new Error('the data length is not divisible by the number of rows');
        }
        super();
        this.rows = rows;
        this.columns = data.length / rows;
        this.data = data;
      }
      set(rowIndex, columnIndex, value) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        this.data[index] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        return this.data[index];
      }
      _calculateIndex(row, column) {
        return row * this.columns + column;
      }
    }
    class WrapperMatrix2D extends AbstractMatrix {
      constructor(data) {
        super();
        this.data = data;
        this.rows = data.length;
        this.columns = data[0].length;
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
    }
    function wrap(array, options) {
      if (isAnyArray.isAnyArray(array)) {
        if (array[0] && isAnyArray.isAnyArray(array[0])) {
          return new WrapperMatrix2D(array);
        } else {
          return new WrapperMatrix1D(array, options);
        }
      } else {
        throw new Error('the argument is not an array');
      }
    }
    class LuDecomposition {
      constructor(matrix) {
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        let lu = matrix.clone();
        let rows = lu.rows;
        let columns = lu.columns;
        let pivotVector = new Float64Array(rows);
        let pivotSign = 1;
        let i, j, k, p, s, t, v;
        let LUcolj, kmax;
        for (i = 0; i < rows; i++) {
          pivotVector[i] = i;
        }
        LUcolj = new Float64Array(rows);
        for (j = 0; j < columns; j++) {
          for (i = 0; i < rows; i++) {
            LUcolj[i] = lu.get(i, j);
          }
          for (i = 0; i < rows; i++) {
            kmax = Math.min(i, j);
            s = 0;
            for (k = 0; k < kmax; k++) {
              s += lu.get(i, k) * LUcolj[k];
            }
            LUcolj[i] -= s;
            lu.set(i, j, LUcolj[i]);
          }
          p = j;
          for (i = j + 1; i < rows; i++) {
            if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
              p = i;
            }
          }
          if (p !== j) {
            for (k = 0; k < columns; k++) {
              t = lu.get(p, k);
              lu.set(p, k, lu.get(j, k));
              lu.set(j, k, t);
            }
            v = pivotVector[p];
            pivotVector[p] = pivotVector[j];
            pivotVector[j] = v;
            pivotSign = -pivotSign;
          }
          if (j < rows && lu.get(j, j) !== 0) {
            for (i = j + 1; i < rows; i++) {
              lu.set(i, j, lu.get(i, j) / lu.get(j, j));
            }
          }
        }
        this.LU = lu;
        this.pivotVector = pivotVector;
        this.pivotSign = pivotSign;
      }
      isSingular() {
        let data = this.LU;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          if (data.get(j, j) === 0) {
            return true;
          }
        }
        return false;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let lu = this.LU;
        let rows = lu.rows;
        if (rows !== value.rows) {
          throw new Error('Invalid matrix dimensions');
        }
        if (this.isSingular()) {
          throw new Error('LU matrix is singular');
        }
        let count = value.columns;
        let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
        let columns = lu.columns;
        let i, j, k;
        for (k = 0; k < columns; k++) {
          for (i = k + 1; i < columns; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        for (k = columns - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / lu.get(k, k));
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        return X;
      }
      get determinant() {
        let data = this.LU;
        if (!data.isSquare()) {
          throw new Error('Matrix must be square');
        }
        let determinant = this.pivotSign;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          determinant *= data.get(j, j);
        }
        return determinant;
      }
      get lowerTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i > j) {
              X.set(i, j, data.get(i, j));
            } else if (i === j) {
              X.set(i, j, 1);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get upperTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$1(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i <= j) {
              X.set(i, j, data.get(i, j));
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get pivotPermutationVector() {
        return Array.from(this.pivotVector);
      }
    }
    function hypotenuse(a, b) {
      let r = 0;
      if (Math.abs(a) > Math.abs(b)) {
        r = b / a;
        return Math.abs(a) * Math.sqrt(1 + r * r);
      }
      if (b !== 0) {
        r = a / b;
        return Math.abs(b) * Math.sqrt(1 + r * r);
      }
      return 0;
    }
    class QrDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let qr = value.clone();
        let m = value.rows;
        let n = value.columns;
        let rdiag = new Float64Array(n);
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          let nrm = 0;
          for (i = k; i < m; i++) {
            nrm = hypotenuse(nrm, qr.get(i, k));
          }
          if (nrm !== 0) {
            if (qr.get(k, k) < 0) {
              nrm = -nrm;
            }
            for (i = k; i < m; i++) {
              qr.set(i, k, qr.get(i, k) / nrm);
            }
            qr.set(k, k, qr.get(k, k) + 1);
            for (j = k + 1; j < n; j++) {
              s = 0;
              for (i = k; i < m; i++) {
                s += qr.get(i, k) * qr.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < m; i++) {
                qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
              }
            }
          }
          rdiag[k] = -nrm;
        }
        this.QR = qr;
        this.Rdiag = rdiag;
      }
      solve(value) {
        value = Matrix$1.checkMatrix(value);
        let qr = this.QR;
        let m = qr.rows;
        if (value.rows !== m) {
          throw new Error('Matrix row dimensions must agree');
        }
        if (!this.isFullRank()) {
          throw new Error('Matrix is rank deficient');
        }
        let count = value.columns;
        let X = value.clone();
        let n = qr.columns;
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          for (j = 0; j < count; j++) {
            s = 0;
            for (i = k; i < m; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }
            s = -s / qr.get(k, k);
            for (i = k; i < m; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
        for (k = n - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / this.Rdiag[k]);
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
            }
          }
        }
        return X.subMatrix(0, n - 1, 0, count - 1);
      }
      isFullRank() {
        let columns = this.QR.columns;
        for (let i = 0; i < columns; i++) {
          if (this.Rdiag[i] === 0) {
            return false;
          }
        }
        return true;
      }
      get upperTriangularMatrix() {
        let qr = this.QR;
        let n = qr.columns;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            if (i < j) {
              X.set(i, j, qr.get(i, j));
            } else if (i === j) {
              X.set(i, j, this.Rdiag[i]);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get orthogonalMatrix() {
        let qr = this.QR;
        let rows = qr.rows;
        let columns = qr.columns;
        let X = new Matrix$1(rows, columns);
        let i, j, k, s;
        for (k = columns - 1; k >= 0; k--) {
          for (i = 0; i < rows; i++) {
            X.set(i, k, 0);
          }
          X.set(k, k, 1);
          for (j = k; j < columns; j++) {
            if (qr.get(k, k) !== 0) {
              s = 0;
              for (i = k; i < rows; i++) {
                s += qr.get(i, k) * X.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < rows; i++) {
                X.set(i, j, X.get(i, j) + s * qr.get(i, k));
              }
            }
          }
        }
        return X;
      }
    }
    class SingularValueDecomposition {
      constructor(value, options = {}) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (value.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let m = value.rows;
        let n = value.columns;
        const {
          computeLeftSingularVectors = true,
          computeRightSingularVectors = true,
          autoTranspose = false
        } = options;
        let wantu = Boolean(computeLeftSingularVectors);
        let wantv = Boolean(computeRightSingularVectors);
        let swapped = false;
        let a;
        if (m < n) {
          if (!autoTranspose) {
            a = value.clone();
            // eslint-disable-next-line no-console
            console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
          } else {
            a = value.transpose();
            m = a.rows;
            n = a.columns;
            swapped = true;
            let aux = wantu;
            wantu = wantv;
            wantv = aux;
          }
        } else {
          a = value.clone();
        }
        let nu = Math.min(m, n);
        let ni = Math.min(m + 1, n);
        let s = new Float64Array(ni);
        let U = new Matrix$1(m, nu);
        let V = new Matrix$1(n, n);
        let e = new Float64Array(n);
        let work = new Float64Array(m);
        let si = new Float64Array(ni);
        for (let i = 0; i < ni; i++) si[i] = i;
        let nct = Math.min(m - 1, n);
        let nrt = Math.max(0, Math.min(n - 2, m));
        let mrc = Math.max(nct, nrt);
        for (let k = 0; k < mrc; k++) {
          if (k < nct) {
            s[k] = 0;
            for (let i = k; i < m; i++) {
              s[k] = hypotenuse(s[k], a.get(i, k));
            }
            if (s[k] !== 0) {
              if (a.get(k, k) < 0) {
                s[k] = -s[k];
              }
              for (let i = k; i < m; i++) {
                a.set(i, k, a.get(i, k) / s[k]);
              }
              a.set(k, k, a.get(k, k) + 1);
            }
            s[k] = -s[k];
          }
          for (let j = k + 1; j < n; j++) {
            if (k < nct && s[k] !== 0) {
              let t = 0;
              for (let i = k; i < m; i++) {
                t += a.get(i, k) * a.get(i, j);
              }
              t = -t / a.get(k, k);
              for (let i = k; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * a.get(i, k));
              }
            }
            e[j] = a.get(k, j);
          }
          if (wantu && k < nct) {
            for (let i = k; i < m; i++) {
              U.set(i, k, a.get(i, k));
            }
          }
          if (k < nrt) {
            e[k] = 0;
            for (let i = k + 1; i < n; i++) {
              e[k] = hypotenuse(e[k], e[i]);
            }
            if (e[k] !== 0) {
              if (e[k + 1] < 0) {
                e[k] = 0 - e[k];
              }
              for (let i = k + 1; i < n; i++) {
                e[i] /= e[k];
              }
              e[k + 1] += 1;
            }
            e[k] = -e[k];
            if (k + 1 < m && e[k] !== 0) {
              for (let i = k + 1; i < m; i++) {
                work[i] = 0;
              }
              for (let i = k + 1; i < m; i++) {
                for (let j = k + 1; j < n; j++) {
                  work[i] += e[j] * a.get(i, j);
                }
              }
              for (let j = k + 1; j < n; j++) {
                let t = -e[j] / e[k + 1];
                for (let i = k + 1; i < m; i++) {
                  a.set(i, j, a.get(i, j) + t * work[i]);
                }
              }
            }
            if (wantv) {
              for (let i = k + 1; i < n; i++) {
                V.set(i, k, e[i]);
              }
            }
          }
        }
        let p = Math.min(n, m + 1);
        if (nct < n) {
          s[nct] = a.get(nct, nct);
        }
        if (m < p) {
          s[p - 1] = 0;
        }
        if (nrt + 1 < p) {
          e[nrt] = a.get(nrt, p - 1);
        }
        e[p - 1] = 0;
        if (wantu) {
          for (let j = nct; j < nu; j++) {
            for (let i = 0; i < m; i++) {
              U.set(i, j, 0);
            }
            U.set(j, j, 1);
          }
          for (let k = nct - 1; k >= 0; k--) {
            if (s[k] !== 0) {
              for (let j = k + 1; j < nu; j++) {
                let t = 0;
                for (let i = k; i < m; i++) {
                  t += U.get(i, k) * U.get(i, j);
                }
                t = -t / U.get(k, k);
                for (let i = k; i < m; i++) {
                  U.set(i, j, U.get(i, j) + t * U.get(i, k));
                }
              }
              for (let i = k; i < m; i++) {
                U.set(i, k, -U.get(i, k));
              }
              U.set(k, k, 1 + U.get(k, k));
              for (let i = 0; i < k - 1; i++) {
                U.set(i, k, 0);
              }
            } else {
              for (let i = 0; i < m; i++) {
                U.set(i, k, 0);
              }
              U.set(k, k, 1);
            }
          }
        }
        if (wantv) {
          for (let k = n - 1; k >= 0; k--) {
            if (k < nrt && e[k] !== 0) {
              for (let j = k + 1; j < n; j++) {
                let t = 0;
                for (let i = k + 1; i < n; i++) {
                  t += V.get(i, k) * V.get(i, j);
                }
                t = -t / V.get(k + 1, k);
                for (let i = k + 1; i < n; i++) {
                  V.set(i, j, V.get(i, j) + t * V.get(i, k));
                }
              }
            }
            for (let i = 0; i < n; i++) {
              V.set(i, k, 0);
            }
            V.set(k, k, 1);
          }
        }
        let pp = p - 1;
        let eps = Number.EPSILON;
        while (p > 0) {
          let k, kase;
          for (k = p - 2; k >= -1; k--) {
            if (k === -1) {
              break;
            }
            const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
            if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
              e[k] = 0;
              break;
            }
          }
          if (k === p - 2) {
            kase = 4;
          } else {
            let ks;
            for (ks = p - 1; ks >= k; ks--) {
              if (ks === k) {
                break;
              }
              let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
              if (Math.abs(s[ks]) <= eps * t) {
                s[ks] = 0;
                break;
              }
            }
            if (ks === k) {
              kase = 3;
            } else if (ks === p - 1) {
              kase = 1;
            } else {
              kase = 2;
              k = ks;
            }
          }
          k++;
          switch (kase) {
            case 1:
              {
                let f = e[p - 2];
                e[p - 2] = 0;
                for (let j = p - 2; j >= k; j--) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  if (j !== k) {
                    f = -sn * e[j - 1];
                    e[j - 1] = cs * e[j - 1];
                  }
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                      V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                      V.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 2:
              {
                let f = e[k - 1];
                e[k - 1] = 0;
                for (let j = k; j < p; j++) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  f = -sn * e[j];
                  e[j] = cs * e[j];
                  if (wantu) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                      U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                      U.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 3:
              {
                const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
                const sp = s[p - 1] / scale;
                const spm1 = s[p - 2] / scale;
                const epm1 = e[p - 2] / scale;
                const sk = s[k] / scale;
                const ek = e[k] / scale;
                const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
                const c = sp * epm1 * (sp * epm1);
                let shift = 0;
                if (b !== 0 || c !== 0) {
                  if (b < 0) {
                    shift = 0 - Math.sqrt(b * b + c);
                  } else {
                    shift = Math.sqrt(b * b + c);
                  }
                  shift = c / (b + shift);
                }
                let f = (sk + sp) * (sk - sp) + shift;
                let g = sk * ek;
                for (let j = k; j < p - 1; j++) {
                  let t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  let cs = f / t;
                  let sn = g / t;
                  if (j !== k) {
                    e[j - 1] = t;
                  }
                  f = cs * s[j] + sn * e[j];
                  e[j] = cs * e[j] - sn * s[j];
                  g = sn * s[j + 1];
                  s[j + 1] = cs * s[j + 1];
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                      V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                      V.set(i, j, t);
                    }
                  }
                  t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  cs = f / t;
                  sn = g / t;
                  s[j] = t;
                  f = cs * e[j] + sn * s[j + 1];
                  s[j + 1] = -sn * e[j] + cs * s[j + 1];
                  g = sn * e[j + 1];
                  e[j + 1] = cs * e[j + 1];
                  if (wantu && j < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                      U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                      U.set(i, j, t);
                    }
                  }
                }
                e[p - 2] = f;
                break;
              }
            case 4:
              {
                if (s[k] <= 0) {
                  s[k] = s[k] < 0 ? -s[k] : 0;
                  if (wantv) {
                    for (let i = 0; i <= pp; i++) {
                      V.set(i, k, -V.get(i, k));
                    }
                  }
                }
                while (k < pp) {
                  if (s[k] >= s[k + 1]) {
                    break;
                  }
                  let t = s[k];
                  s[k] = s[k + 1];
                  s[k + 1] = t;
                  if (wantv && k < n - 1) {
                    for (let i = 0; i < n; i++) {
                      t = V.get(i, k + 1);
                      V.set(i, k + 1, V.get(i, k));
                      V.set(i, k, t);
                    }
                  }
                  if (wantu && k < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = U.get(i, k + 1);
                      U.set(i, k + 1, U.get(i, k));
                      U.set(i, k, t);
                    }
                  }
                  k++;
                }
                p--;
                break;
              }
            // no default
          }
        }
        if (swapped) {
          let tmp = V;
          V = U;
          U = tmp;
        }
        this.m = m;
        this.n = n;
        this.s = s;
        this.U = U;
        this.V = V;
      }
      solve(value) {
        let Y = value;
        let e = this.threshold;
        let scols = this.s.length;
        let Ls = Matrix$1.zeros(scols, scols);
        for (let i = 0; i < scols; i++) {
          if (Math.abs(this.s[i]) <= e) {
            Ls.set(i, i, 0);
          } else {
            Ls.set(i, i, 1 / this.s[i]);
          }
        }
        let U = this.U;
        let V = this.rightSingularVectors;
        let VL = V.mmul(Ls);
        let vrows = V.rows;
        let urows = U.rows;
        let VLU = Matrix$1.zeros(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < scols; k++) {
              sum += VL.get(i, k) * U.get(j, k);
            }
            VLU.set(i, j, sum);
          }
        }
        return VLU.mmul(Y);
      }
      solveForDiagonal(value) {
        return this.solve(Matrix$1.diag(value));
      }
      inverse() {
        let V = this.V;
        let e = this.threshold;
        let vrows = V.rows;
        let vcols = V.columns;
        let X = new Matrix$1(vrows, this.s.length);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < vcols; j++) {
            if (Math.abs(this.s[j]) > e) {
              X.set(i, j, V.get(i, j) / this.s[j]);
            }
          }
        }
        let U = this.U;
        let urows = U.rows;
        let ucols = U.columns;
        let Y = new Matrix$1(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < ucols; k++) {
              sum += X.get(i, k) * U.get(j, k);
            }
            Y.set(i, j, sum);
          }
        }
        return Y;
      }
      get condition() {
        return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
      }
      get norm2() {
        return this.s[0];
      }
      get rank() {
        let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
        let r = 0;
        let s = this.s;
        for (let i = 0, ii = s.length; i < ii; i++) {
          if (s[i] > tol) {
            r++;
          }
        }
        return r;
      }
      get diagonal() {
        return Array.from(this.s);
      }
      get threshold() {
        return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
      }
      get leftSingularVectors() {
        return this.U;
      }
      get rightSingularVectors() {
        return this.V;
      }
      get diagonalMatrix() {
        return Matrix$1.diag(this.s);
      }
    }
    function inverse(matrix, useSVD = false) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      if (useSVD) {
        return new SingularValueDecomposition(matrix).inverse();
      } else {
        return solve(matrix, Matrix$1.eye(matrix.rows));
      }
    }
    function solve(leftHandSide, rightHandSide, useSVD = false) {
      leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
      rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
      if (useSVD) {
        return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
      } else {
        return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
      }
    }
    function determinant(matrix) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isSquare()) {
        if (matrix.columns === 0) {
          return 1;
        }
        let a, b, c, d;
        if (matrix.columns === 2) {
          // 2 x 2 matrix
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(1, 0);
          d = matrix.get(1, 1);
          return a * d - b * c;
        } else if (matrix.columns === 3) {
          // 3 x 3 matrix
          let subMatrix0, subMatrix1, subMatrix2;
          subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
          subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
          subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(0, 2);
          return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
        } else {
          // general purpose determinant using the LU decomposition
          return new LuDecomposition(matrix).determinant;
        }
      } else {
        throw Error('determinant can only be calculated for a square matrix');
      }
    }
    function xrange(n, exception) {
      let range = [];
      for (let i = 0; i < n; i++) {
        if (i !== exception) {
          range.push(i);
        }
      }
      return range;
    }
    function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
      if (error > thresholdError) {
        return new Array(matrix.rows + 1).fill(0);
      } else {
        let returnArray = matrix.addRow(index, [0]);
        for (let i = 0; i < returnArray.rows; i++) {
          if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
            returnArray.set(i, 0, 0);
          }
        }
        return returnArray.to1DArray();
      }
    }
    function linearDependencies(matrix, options = {}) {
      const {
        thresholdValue = 10e-10,
        thresholdError = 10e-10
      } = options;
      matrix = Matrix$1.checkMatrix(matrix);
      let n = matrix.rows;
      let results = new Matrix$1(n, n);
      for (let i = 0; i < n; i++) {
        let b = Matrix$1.columnVector(matrix.getRow(i));
        let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
        let svd = new SingularValueDecomposition(Abis);
        let x = svd.solve(b);
        let error = Matrix$1.sub(b, Abis.mmul(x)).abs().max();
        results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
      }
      return results;
    }
    function pseudoInverse(matrix, threshold = Number.EPSILON) {
      matrix = Matrix$1.checkMatrix(matrix);
      if (matrix.isEmpty()) {
        // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
        // (0xn)*(nx0)*(0xn) = 0xn
        // (nx0)*(0xn)*(nx0) = nx0
        return matrix.transpose();
      }
      let svdSolution = new SingularValueDecomposition(matrix, {
        autoTranspose: true
      });
      let U = svdSolution.leftSingularVectors;
      let V = svdSolution.rightSingularVectors;
      let s = svdSolution.diagonal;
      for (let i = 0; i < s.length; i++) {
        if (Math.abs(s[i]) > threshold) {
          s[i] = 1.0 / s[i];
        } else {
          s[i] = 0.0;
        }
      }
      return V.mmul(Matrix$1.diag(s).mmul(U.transpose()));
    }
    function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true
      } = options;
      if (center) {
        xMatrix = xMatrix.center('column');
        if (!yIsSame) {
          yMatrix = yMatrix.center('column');
        }
      }
      const cov = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < cov.rows; i++) {
        for (let j = 0; j < cov.columns; j++) {
          cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
        }
      }
      return cov;
    }
    function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$1(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$1(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true,
        scale = true
      } = options;
      if (center) {
        xMatrix.center('column');
        if (!yIsSame) {
          yMatrix.center('column');
        }
      }
      if (scale) {
        xMatrix.scale('column');
        if (!yIsSame) {
          yMatrix.scale('column');
        }
      }
      const sdx = xMatrix.standardDeviation('column', {
        unbiased: true
      });
      const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
        unbiased: true
      });
      const corr = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < corr.rows; i++) {
        for (let j = 0; j < corr.columns; j++) {
          corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
        }
      }
      return corr;
    }
    class EigenvalueDecomposition {
      constructor(matrix, options = {}) {
        const {
          assumeSymmetric = false
        } = options;
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        if (!matrix.isSquare()) {
          throw new Error('Matrix is not a square matrix');
        }
        if (matrix.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let n = matrix.columns;
        let V = new Matrix$1(n, n);
        let d = new Float64Array(n);
        let e = new Float64Array(n);
        let value = matrix;
        let i, j;
        let isSymmetric = false;
        if (assumeSymmetric) {
          isSymmetric = true;
        } else {
          isSymmetric = matrix.isSymmetric();
        }
        if (isSymmetric) {
          for (i = 0; i < n; i++) {
            for (j = 0; j < n; j++) {
              V.set(i, j, value.get(i, j));
            }
          }
          tred2(n, e, d, V);
          tql2(n, e, d, V);
        } else {
          let H = new Matrix$1(n, n);
          let ort = new Float64Array(n);
          for (j = 0; j < n; j++) {
            for (i = 0; i < n; i++) {
              H.set(i, j, value.get(i, j));
            }
          }
          orthes(n, H, ort, V);
          hqr2(n, e, d, V, H);
        }
        this.n = n;
        this.e = e;
        this.d = d;
        this.V = V;
      }
      get realEigenvalues() {
        return Array.from(this.d);
      }
      get imaginaryEigenvalues() {
        return Array.from(this.e);
      }
      get eigenvectorMatrix() {
        return this.V;
      }
      get diagonalMatrix() {
        let n = this.n;
        let e = this.e;
        let d = this.d;
        let X = new Matrix$1(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            X.set(i, j, 0);
          }
          X.set(i, i, d[i]);
          if (e[i] > 0) {
            X.set(i, i + 1, e[i]);
          } else if (e[i] < 0) {
            X.set(i, i - 1, e[i]);
          }
        }
        return X;
      }
    }
    function tred2(n, e, d, V) {
      let f, g, h, i, j, k, hh, scale;
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
      }
      for (i = n - 1; i > 0; i--) {
        scale = 0;
        h = 0;
        for (k = 0; k < i; k++) {
          scale = scale + Math.abs(d[k]);
        }
        if (scale === 0) {
          e[i] = d[i - 1];
          for (j = 0; j < i; j++) {
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
            V.set(j, i, 0);
          }
        } else {
          for (k = 0; k < i; k++) {
            d[k] /= scale;
            h += d[k] * d[k];
          }
          f = d[i - 1];
          g = Math.sqrt(h);
          if (f > 0) {
            g = -g;
          }
          e[i] = scale * g;
          h = h - f * g;
          d[i - 1] = f - g;
          for (j = 0; j < i; j++) {
            e[j] = 0;
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            V.set(j, i, f);
            g = e[j] + V.get(j, j) * f;
            for (k = j + 1; k <= i - 1; k++) {
              g += V.get(k, j) * d[k];
              e[k] += V.get(k, j) * f;
            }
            e[j] = g;
          }
          f = 0;
          for (j = 0; j < i; j++) {
            e[j] /= h;
            f += e[j] * d[j];
          }
          hh = f / (h + h);
          for (j = 0; j < i; j++) {
            e[j] -= hh * d[j];
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            g = e[j];
            for (k = j; k <= i - 1; k++) {
              V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
            }
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
          }
        }
        d[i] = h;
      }
      for (i = 0; i < n - 1; i++) {
        V.set(n - 1, i, V.get(i, i));
        V.set(i, i, 1);
        h = d[i + 1];
        if (h !== 0) {
          for (k = 0; k <= i; k++) {
            d[k] = V.get(k, i + 1) / h;
          }
          for (j = 0; j <= i; j++) {
            g = 0;
            for (k = 0; k <= i; k++) {
              g += V.get(k, i + 1) * V.get(k, j);
            }
            for (k = 0; k <= i; k++) {
              V.set(k, j, V.get(k, j) - g * d[k]);
            }
          }
        }
        for (k = 0; k <= i; k++) {
          V.set(k, i + 1, 0);
        }
      }
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
        V.set(n - 1, j, 0);
      }
      V.set(n - 1, n - 1, 1);
      e[0] = 0;
    }
    function tql2(n, e, d, V) {
      let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
      for (i = 1; i < n; i++) {
        e[i - 1] = e[i];
      }
      e[n - 1] = 0;
      let f = 0;
      let tst1 = 0;
      let eps = Number.EPSILON;
      for (l = 0; l < n; l++) {
        tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
        m = l;
        while (m < n) {
          if (Math.abs(e[m]) <= eps * tst1) {
            break;
          }
          m++;
        }
        if (m > l) {
          do {
            g = d[l];
            p = (d[l + 1] - g) / (2 * e[l]);
            r = hypotenuse(p, 1);
            if (p < 0) {
              r = -r;
            }
            d[l] = e[l] / (p + r);
            d[l + 1] = e[l] * (p + r);
            dl1 = d[l + 1];
            h = g - d[l];
            for (i = l + 2; i < n; i++) {
              d[i] -= h;
            }
            f = f + h;
            p = d[m];
            c = 1;
            c2 = c;
            c3 = c;
            el1 = e[l + 1];
            s = 0;
            s2 = 0;
            for (i = m - 1; i >= l; i--) {
              c3 = c2;
              c2 = c;
              s2 = s;
              g = c * e[i];
              h = c * p;
              r = hypotenuse(p, e[i]);
              e[i + 1] = s * r;
              s = e[i] / r;
              c = p / r;
              p = c * d[i] - s * g;
              d[i + 1] = h + s * (c * g + s * d[i]);
              for (k = 0; k < n; k++) {
                h = V.get(k, i + 1);
                V.set(k, i + 1, s * V.get(k, i) + c * h);
                V.set(k, i, c * V.get(k, i) - s * h);
              }
            }
            p = -s * s2 * c3 * el1 * e[l] / dl1;
            e[l] = s * p;
            d[l] = c * p;
          } while (Math.abs(e[l]) > eps * tst1);
        }
        d[l] = d[l] + f;
        e[l] = 0;
      }
      for (i = 0; i < n - 1; i++) {
        k = i;
        p = d[i];
        for (j = i + 1; j < n; j++) {
          if (d[j] < p) {
            k = j;
            p = d[j];
          }
        }
        if (k !== i) {
          d[k] = d[i];
          d[i] = p;
          for (j = 0; j < n; j++) {
            p = V.get(j, i);
            V.set(j, i, V.get(j, k));
            V.set(j, k, p);
          }
        }
      }
    }
    function orthes(n, H, ort, V) {
      let low = 0;
      let high = n - 1;
      let f, g, h, i, j, m;
      let scale;
      for (m = low + 1; m <= high - 1; m++) {
        scale = 0;
        for (i = m; i <= high; i++) {
          scale = scale + Math.abs(H.get(i, m - 1));
        }
        if (scale !== 0) {
          h = 0;
          for (i = high; i >= m; i--) {
            ort[i] = H.get(i, m - 1) / scale;
            h += ort[i] * ort[i];
          }
          g = Math.sqrt(h);
          if (ort[m] > 0) {
            g = -g;
          }
          h = h - ort[m] * g;
          ort[m] = ort[m] - g;
          for (j = m; j < n; j++) {
            f = 0;
            for (i = high; i >= m; i--) {
              f += ort[i] * H.get(i, j);
            }
            f = f / h;
            for (i = m; i <= high; i++) {
              H.set(i, j, H.get(i, j) - f * ort[i]);
            }
          }
          for (i = 0; i <= high; i++) {
            f = 0;
            for (j = high; j >= m; j--) {
              f += ort[j] * H.get(i, j);
            }
            f = f / h;
            for (j = m; j <= high; j++) {
              H.set(i, j, H.get(i, j) - f * ort[j]);
            }
          }
          ort[m] = scale * ort[m];
          H.set(m, m - 1, scale * g);
        }
      }
      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          V.set(i, j, i === j ? 1 : 0);
        }
      }
      for (m = high - 1; m >= low + 1; m--) {
        if (H.get(m, m - 1) !== 0) {
          for (i = m + 1; i <= high; i++) {
            ort[i] = H.get(i, m - 1);
          }
          for (j = m; j <= high; j++) {
            g = 0;
            for (i = m; i <= high; i++) {
              g += ort[i] * V.get(i, j);
            }
            g = g / ort[m] / H.get(m, m - 1);
            for (i = m; i <= high; i++) {
              V.set(i, j, V.get(i, j) + g * ort[i]);
            }
          }
        }
      }
    }
    function hqr2(nn, e, d, V, H) {
      let n = nn - 1;
      let low = 0;
      let high = nn - 1;
      let eps = Number.EPSILON;
      let exshift = 0;
      let norm = 0;
      let p = 0;
      let q = 0;
      let r = 0;
      let s = 0;
      let z = 0;
      let iter = 0;
      let i, j, k, l, m, t, w, x, y;
      let ra, sa, vr, vi;
      let notlast, cdivres;
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          d[i] = H.get(i, i);
          e[i] = 0;
        }
        for (j = Math.max(i - 1, 0); j < nn; j++) {
          norm = norm + Math.abs(H.get(i, j));
        }
      }
      while (n >= low) {
        l = n;
        while (l > low) {
          s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
          if (s === 0) {
            s = norm;
          }
          if (Math.abs(H.get(l, l - 1)) < eps * s) {
            break;
          }
          l--;
        }
        if (l === n) {
          H.set(n, n, H.get(n, n) + exshift);
          d[n] = H.get(n, n);
          e[n] = 0;
          n--;
          iter = 0;
        } else if (l === n - 1) {
          w = H.get(n, n - 1) * H.get(n - 1, n);
          p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
          q = p * p + w;
          z = Math.sqrt(Math.abs(q));
          H.set(n, n, H.get(n, n) + exshift);
          H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
          x = H.get(n, n);
          if (q >= 0) {
            z = p >= 0 ? p + z : p - z;
            d[n - 1] = x + z;
            d[n] = d[n - 1];
            if (z !== 0) {
              d[n] = x - w / z;
            }
            e[n - 1] = 0;
            e[n] = 0;
            x = H.get(n, n - 1);
            s = Math.abs(x) + Math.abs(z);
            p = x / s;
            q = z / s;
            r = Math.sqrt(p * p + q * q);
            p = p / r;
            q = q / r;
            for (j = n - 1; j < nn; j++) {
              z = H.get(n - 1, j);
              H.set(n - 1, j, q * z + p * H.get(n, j));
              H.set(n, j, q * H.get(n, j) - p * z);
            }
            for (i = 0; i <= n; i++) {
              z = H.get(i, n - 1);
              H.set(i, n - 1, q * z + p * H.get(i, n));
              H.set(i, n, q * H.get(i, n) - p * z);
            }
            for (i = low; i <= high; i++) {
              z = V.get(i, n - 1);
              V.set(i, n - 1, q * z + p * V.get(i, n));
              V.set(i, n, q * V.get(i, n) - p * z);
            }
          } else {
            d[n - 1] = x + p;
            d[n] = x + p;
            e[n - 1] = z;
            e[n] = -z;
          }
          n = n - 2;
          iter = 0;
        } else {
          x = H.get(n, n);
          y = 0;
          w = 0;
          if (l < n) {
            y = H.get(n - 1, n - 1);
            w = H.get(n, n - 1) * H.get(n - 1, n);
          }
          if (iter === 10) {
            exshift += x;
            for (i = low; i <= n; i++) {
              H.set(i, i, H.get(i, i) - x);
            }
            s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
            // eslint-disable-next-line no-multi-assign
            x = y = 0.75 * s;
            w = -0.4375 * s * s;
          }
          if (iter === 30) {
            s = (y - x) / 2;
            s = s * s + w;
            if (s > 0) {
              s = Math.sqrt(s);
              if (y < x) {
                s = -s;
              }
              s = x - w / ((y - x) / 2 + s);
              for (i = low; i <= n; i++) {
                H.set(i, i, H.get(i, i) - s);
              }
              exshift += s;
              // eslint-disable-next-line no-multi-assign
              x = y = w = 0.964;
            }
          }
          iter = iter + 1;
          m = n - 2;
          while (m >= l) {
            z = H.get(m, m);
            r = x - z;
            s = y - z;
            p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
            q = H.get(m + 1, m + 1) - z - r - s;
            r = H.get(m + 2, m + 1);
            s = Math.abs(p) + Math.abs(q) + Math.abs(r);
            p = p / s;
            q = q / s;
            r = r / s;
            if (m === l) {
              break;
            }
            if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
              break;
            }
            m--;
          }
          for (i = m + 2; i <= n; i++) {
            H.set(i, i - 2, 0);
            if (i > m + 2) {
              H.set(i, i - 3, 0);
            }
          }
          for (k = m; k <= n - 1; k++) {
            notlast = k !== n - 1;
            if (k !== m) {
              p = H.get(k, k - 1);
              q = H.get(k + 1, k - 1);
              r = notlast ? H.get(k + 2, k - 1) : 0;
              x = Math.abs(p) + Math.abs(q) + Math.abs(r);
              if (x !== 0) {
                p = p / x;
                q = q / x;
                r = r / x;
              }
            }
            if (x === 0) {
              break;
            }
            s = Math.sqrt(p * p + q * q + r * r);
            if (p < 0) {
              s = -s;
            }
            if (s !== 0) {
              if (k !== m) {
                H.set(k, k - 1, -s * x);
              } else if (l !== m) {
                H.set(k, k - 1, -H.get(k, k - 1));
              }
              p = p + s;
              x = p / s;
              y = q / s;
              z = r / s;
              q = q / p;
              r = r / p;
              for (j = k; j < nn; j++) {
                p = H.get(k, j) + q * H.get(k + 1, j);
                if (notlast) {
                  p = p + r * H.get(k + 2, j);
                  H.set(k + 2, j, H.get(k + 2, j) - p * z);
                }
                H.set(k, j, H.get(k, j) - p * x);
                H.set(k + 1, j, H.get(k + 1, j) - p * y);
              }
              for (i = 0; i <= Math.min(n, k + 3); i++) {
                p = x * H.get(i, k) + y * H.get(i, k + 1);
                if (notlast) {
                  p = p + z * H.get(i, k + 2);
                  H.set(i, k + 2, H.get(i, k + 2) - p * r);
                }
                H.set(i, k, H.get(i, k) - p);
                H.set(i, k + 1, H.get(i, k + 1) - p * q);
              }
              for (i = low; i <= high; i++) {
                p = x * V.get(i, k) + y * V.get(i, k + 1);
                if (notlast) {
                  p = p + z * V.get(i, k + 2);
                  V.set(i, k + 2, V.get(i, k + 2) - p * r);
                }
                V.set(i, k, V.get(i, k) - p);
                V.set(i, k + 1, V.get(i, k + 1) - p * q);
              }
            }
          }
        }
      }
      if (norm === 0) {
        return;
      }
      for (n = nn - 1; n >= 0; n--) {
        p = d[n];
        q = e[n];
        if (q === 0) {
          l = n;
          H.set(n, n, 1);
          for (i = n - 1; i >= 0; i--) {
            w = H.get(i, i) - p;
            r = 0;
            for (j = l; j <= n; j++) {
              r = r + H.get(i, j) * H.get(j, n);
            }
            if (e[i] < 0) {
              z = w;
              s = r;
            } else {
              l = i;
              if (e[i] === 0) {
                H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
                t = (x * s - z * r) / q;
                H.set(i, n, t);
                H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
              }
              t = Math.abs(H.get(i, n));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        } else if (q < 0) {
          l = n - 1;
          if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
            H.set(n - 1, n - 1, q / H.get(n, n - 1));
            H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
          } else {
            cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
            H.set(n - 1, n - 1, cdivres[0]);
            H.set(n - 1, n, cdivres[1]);
          }
          H.set(n, n - 1, 0);
          H.set(n, n, 1);
          for (i = n - 2; i >= 0; i--) {
            ra = 0;
            sa = 0;
            for (j = l; j <= n; j++) {
              ra = ra + H.get(i, j) * H.get(j, n - 1);
              sa = sa + H.get(i, j) * H.get(j, n);
            }
            w = H.get(i, i) - p;
            if (e[i] < 0) {
              z = w;
              r = ra;
              s = sa;
            } else {
              l = i;
              if (e[i] === 0) {
                cdivres = cdiv(-ra, -sa, w, q);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
                vi = (d[i] - p) * 2 * q;
                if (vr === 0 && vi === 0) {
                  vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
                }
                cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
                if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
                  H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
                  H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
                } else {
                  cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
                  H.set(i + 1, n - 1, cdivres[0]);
                  H.set(i + 1, n, cdivres[1]);
                }
              }
              t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n - 1, H.get(j, n - 1) / t);
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        }
      }
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          for (j = i; j < nn; j++) {
            V.set(i, j, H.get(i, j));
          }
        }
      }
      for (j = nn - 1; j >= low; j--) {
        for (i = low; i <= high; i++) {
          z = 0;
          for (k = low; k <= Math.min(j, high); k++) {
            z = z + V.get(i, k) * H.get(k, j);
          }
          V.set(i, j, z);
        }
      }
    }
    function cdiv(xr, xi, yr, yi) {
      let r, d;
      if (Math.abs(yr) > Math.abs(yi)) {
        r = yi / yr;
        d = yr + r * yi;
        return [(xr + r * xi) / d, (xi - r * xr) / d];
      } else {
        r = yr / yi;
        d = yi + r * yr;
        return [(r * xr + xi) / d, (r * xi - xr) / d];
      }
    }
    class CholeskyDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (!value.isSymmetric()) {
          throw new Error('Matrix is not symmetric');
        }
        let a = value;
        let dimension = a.rows;
        let l = new Matrix$1(dimension, dimension);
        let positiveDefinite = true;
        let i, j, k;
        for (j = 0; j < dimension; j++) {
          let d = 0;
          for (k = 0; k < j; k++) {
            let s = 0;
            for (i = 0; i < k; i++) {
              s += l.get(k, i) * l.get(j, i);
            }
            s = (a.get(j, k) - s) / l.get(k, k);
            l.set(j, k, s);
            d = d + s * s;
          }
          d = a.get(j, j) - d;
          positiveDefinite &&= d > 0;
          l.set(j, j, Math.sqrt(Math.max(d, 0)));
          for (k = j + 1; k < dimension; k++) {
            l.set(j, k, 0);
          }
        }
        this.L = l;
        this.positiveDefinite = positiveDefinite;
      }
      isPositiveDefinite() {
        return this.positiveDefinite;
      }
      solve(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let l = this.L;
        let dimension = l.rows;
        if (value.rows !== dimension) {
          throw new Error('Matrix dimensions do not match');
        }
        if (this.isPositiveDefinite() === false) {
          throw new Error('Matrix is not positive definite');
        }
        let count = value.columns;
        let B = value.clone();
        let i, j, k;
        for (k = 0; k < dimension; k++) {
          for (j = 0; j < count; j++) {
            for (i = 0; i < k; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        for (k = dimension - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            for (i = k + 1; i < dimension; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        return B;
      }
      get lowerTriangularMatrix() {
        return this.L;
      }
    }
    class nipals {
      constructor(X, options = {}) {
        X = WrapperMatrix2D.checkMatrix(X);
        let {
          Y
        } = options;
        const {
          scaleScores = false,
          maxIterations = 1000,
          terminationCriteria = 1e-10
        } = options;
        let u;
        if (Y) {
          if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
            Y = Matrix$1.columnVector(Y);
          } else {
            Y = WrapperMatrix2D.checkMatrix(Y);
          }
          if (Y.rows !== X.rows) {
            throw new Error('Y should have the same number of rows as X');
          }
          u = Y.getColumnVector(0);
        } else {
          u = X.getColumnVector(0);
        }
        let diff = 1;
        let t, q, w, tOld;
        for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
          w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
          w = w.div(w.norm());
          t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
          if (counter > 0) {
            diff = t.clone().sub(tOld).pow(2).sum();
          }
          tOld = t.clone();
          if (Y) {
            q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
            q = q.div(q.norm());
            u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
          } else {
            u = t;
          }
        }
        if (Y) {
          let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          p = p.div(p.norm());
          let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
          let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
          this.t = t;
          this.p = p.transpose();
          this.w = w.transpose();
          this.q = q;
          this.u = u;
          this.s = t.transpose().mmul(t);
          this.xResidual = xResidual;
          this.yResidual = yResidual;
          this.betas = residual;
        } else {
          this.w = w.transpose();
          this.s = t.transpose().mmul(t).sqrt();
          if (scaleScores) {
            this.t = t.clone().div(this.s.get(0, 0));
          } else {
            this.t = t;
          }
          this.xResidual = X.sub(t.mmul(w.transpose()));
        }
      }
    }
    matrix.AbstractMatrix = AbstractMatrix;
    matrix.CHO = CholeskyDecomposition;
    matrix.CholeskyDecomposition = CholeskyDecomposition;
    matrix.DistanceMatrix = DistanceMatrix;
    matrix.EVD = EigenvalueDecomposition;
    matrix.EigenvalueDecomposition = EigenvalueDecomposition;
    matrix.LU = LuDecomposition;
    matrix.LuDecomposition = LuDecomposition;
    var Matrix_1 = matrix.Matrix = Matrix$1;
    matrix.MatrixColumnSelectionView = MatrixColumnSelectionView;
    matrix.MatrixColumnView = MatrixColumnView;
    matrix.MatrixFlipColumnView = MatrixFlipColumnView;
    matrix.MatrixFlipRowView = MatrixFlipRowView;
    matrix.MatrixRowSelectionView = MatrixRowSelectionView;
    matrix.MatrixRowView = MatrixRowView;
    matrix.MatrixSelectionView = MatrixSelectionView;
    matrix.MatrixSubView = MatrixSubView;
    matrix.MatrixTransposeView = MatrixTransposeView;
    matrix.NIPALS = nipals;
    matrix.Nipals = nipals;
    matrix.QR = QrDecomposition;
    matrix.QrDecomposition = QrDecomposition;
    matrix.SVD = SingularValueDecomposition;
    matrix.SingularValueDecomposition = SingularValueDecomposition;
    matrix.SymmetricMatrix = SymmetricMatrix;
    matrix.WrapperMatrix1D = WrapperMatrix1D;
    matrix.WrapperMatrix2D = WrapperMatrix2D;
    matrix.correlation = correlation;
    matrix.covariance = covariance;
    var _default = matrix.default = Matrix$1;
    matrix.determinant = determinant;
    matrix.inverse = inverse;
    matrix.linearDependencies = linearDependencies;
    matrix.pseudoInverse = pseudoInverse;
    matrix.solve = solve;
    matrix.wrap = wrap;

    const Matrix = Matrix_1;
    _default.Matrix ? _default.Matrix : Matrix_1;

    /**
     * Algorithm that finds the shortest distance from one node to the other
     * @param {Matrix} adjMatrix - A squared adjacency matrix
     * @return {Matrix} - Distance from a node to the other, -1 if the node is unreachable
     */
    function floydWarshall(adjMatrix) {
      if (Matrix.isMatrix(adjMatrix) && adjMatrix.columns !== adjMatrix.rows) {
        throw new TypeError('The adjacency matrix should be squared');
      }
      const numVertices = adjMatrix.columns;
      let distMatrix = new Matrix(numVertices, numVertices);
      distMatrix.apply((row, column) => {
        // principal diagonal is 0
        if (row === column) {
          distMatrix.set(row, column, 0);
        } else {
          let val = adjMatrix.get(row, column);
          if (val || Object.is(val, -0)) {
            // edges values remain the same
            distMatrix.set(row, column, val);
          } else {
            // 0 values become infinity
            distMatrix.set(row, column, Number.POSITIVE_INFINITY);
          }
        }
      });
      for (let k = 0; k < numVertices; ++k) {
        for (let i = 0; i < numVertices; ++i) {
          for (let j = 0; j < numVertices; ++j) {
            let dist = distMatrix.get(i, k) + distMatrix.get(k, j);
            if (distMatrix.get(i, j) > dist) {
              distMatrix.set(i, j, dist);
            }
          }
        }
      }
      // When there's no connection the value is -1
      distMatrix.apply((row, column) => {
        if (distMatrix.get(row, column) === Number.POSITIVE_INFINITY) {
          distMatrix.set(row, column, -1);
        }
      });
      return distMatrix;
    }

    /**
     * Returns a connectivity matrix
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {boolean} [options.pathLength=false] - get the path length between atoms
     * @param {boolean} [options.mass=false] - set the nominal mass of the atoms on diagonal
     * @param {boolean} [options.atomicNo=false] - set the atomic number of the atom on diagonal
     * @param {boolean} [options.negativeAtomicNo=false] - set the atomic number * -1 of the atom on diagonal
     * @param {boolean} [options.sdt=false] - set 1, 2 or 3 depending if single, double or triple bond
     * @param {boolean} [options.sdta=false] - set 1, 2, 3 or 4 depending if single, double, triple or aromatic  bond
     */
    function getConnectivityMatrix(molecule, options = {}) {
      const OCL = molecule.getOCL();
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      const nbAtoms = molecule.getAllAtoms();
      let result = new Array(nbAtoms).fill();
      result = result.map(() => new Array(nbAtoms).fill(0));
      if (!options.pathLength) {
        if (options.atomicNo) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = molecule.getAtomicNo(i);
          }
        } else if (options.negativeAtomicNo) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = -molecule.getAtomicNo(i);
          }
        } else if (options.mass) {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = OCL.Molecule.cRoundedMass[molecule.getAtomicNo(i)];
          }
        } else {
          for (let i = 0; i < nbAtoms; i++) {
            result[i][i] = 1;
          }
        }
      }
      if (options.sdt) {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
          }
        }
      } else if (options.sdta) {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            const bondNumber = molecule.getConnBond(i, j);
            if (molecule.isAromaticBond(bondNumber)) {
              result[i][molecule.getConnAtom(i, j)] = 4;
            } else {
              result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
            }
          }
        }
      } else {
        for (let i = 0; i < nbAtoms; i++) {
          const l = molecule.getAllConnAtoms(i);
          for (let j = 0; j < l; j++) {
            result[i][molecule.getConnAtom(i, j)] = 1;
          }
        }
      }
      if (options.pathLength) {
        result = floydWarshall(new Matrix(result)).to2DArray();
      }
      return result;
    }

    /**
     * Creates a compact copy of the molecule without custom labels.
     * We don't want that custom labels interfere with hose code and diaID generation.
     * @param molecule
     * @returns
     */
    function getCompactCopyWithoutCustomLabels(molecule) {
      const tempMolecule = molecule.getCompactCopy();
      for (let i = 0; i < tempMolecule.getAllAtoms(); i++) {
        tempMolecule.setAtomCustomLabel(i, '');
      }
      return tempMolecule;
    }

    function getCanonizedDiaIDs(diaMol, options) {
      const {
        logger,
        maxNbAtoms
      } = options;
      const moleculeWithH = diaMol.moleculeWithH;
      if (moleculeWithH.getAllAtoms() > maxNbAtoms) {
        logger.warn(`too many atoms to evaluate heterotopic chiral bonds: ${moleculeWithH.getAllAtoms()} > ${maxNbAtoms}`);
        return [];
      }
      const heterotopicSymmetryRanks = diaMol.heterotopicSymmetryRanks;
      const finalRanks = diaMol.finalRanks;
      const canonizedDiaIDs = new Array(moleculeWithH.getAllAtoms());
      moleculeWithH.ensureHelperArrays(diaMol.molecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const cache = {};
      for (let i = 0; i < diaMol.moleculeWithH.getAllAtoms(); i++) {
        const rank = heterotopicSymmetryRanks[i];
        if (rank && cache[rank]) {
          canonizedDiaIDs[finalRanks[i]] = cache[rank].diaID;
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(diaMol.moleculeWithH);
        tagAtom(tempMolecule, i);
        makeRacemic(tempMolecule);
        const diaID = tempMolecule.getCanonizedIDCode(diaMol.molecule.getOCL().Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        canonizedDiaIDs[finalRanks[i]] = diaID;
      }
      return canonizedDiaIDs;
    }

    /**
     * Returns an array of strings (idCodes) specified molecule. Each string corresponds to a
     * hose code. By default it will calculate the hose codes for sphere 0 to 4 and will reuse
     * the existing tagged atoms.
     * @param molecule - The OCL molecule to process.
     * @param options - Options for generating hose codes.
     * @returns An array of hose code strings.
     */
    function getHoseCodesForAtomsAsStrings(molecule, options = {}) {
      const fragments = getHoseCodesForAtomsAsFragments(molecule, options);
      const OCL = molecule.getOCL();
      const hoses = [];
      for (const fragment of fragments) {
        hoses.push(fragment.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS));
      }
      return hoses;
    }

    /**
     * Get the canonized hose codes for a topic molecule. It will use the moleculeWithH
     * @param topicMolecule - The topic molecule to get the hose codes for.
     * @returns The canonized hose codes.
     */
    function getCanonizedHoseCodes(topicMolecule) {
      const options = topicMolecule.options;
      const heterotopicSymmetryRanks = topicMolecule.heterotopicSymmetryRanks;
      const moleculeWithH = topicMolecule.moleculeWithH;
      const finalRanks = topicMolecule.finalRanks;
      const canonizedHoseCodes = new Array(moleculeWithH.getAllAtoms());
      moleculeWithH.ensureHelperArrays(topicMolecule.molecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const cache = {};
      for (let i = 0; i < topicMolecule.moleculeWithH.getAllAtoms(); i++) {
        const rank = heterotopicSymmetryRanks[i];
        if (rank && cache[rank]) {
          canonizedHoseCodes[finalRanks[i]] = cache[rank].diaID;
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(moleculeWithH);
        tagAtom(tempMolecule, i);
        const hoses = getHoseCodesForAtomsAsStrings(tempMolecule, options);
        canonizedHoseCodes[finalRanks[i]] = hoses;
      }
      return canonizedHoseCodes;
    }

    /**
     * For each atom we will return an array of objects
     * @param topicMolecule
     * @param options
     * @returns
     */
    function getCanonizedHoseCodesForPath(topicMolecule, options = {}) {
      const {
        minPathLength = 0,
        maxPathLength = topicMolecule.options.maxPathLength,
        toAtomicNo,
        fromAtomicNo
      } = options;
      if (maxPathLength > topicMolecule.options.maxPathLength) {
        throw new Error(`maxPathLength cannot be larger than the one defined in topicMolecule: ${topicMolecule.options.maxPathLength}`);
      }
      const atomsPaths = topicMolecule.atomsPaths;
      const molecule = topicMolecule.moleculeWithH;
      const results = new Array(molecule.getAllAtoms());
      for (let fromAtom = 0; fromAtom < molecule.getAllAtoms(); fromAtom++) {
        results[fromAtom] = {
          fromDiaID: topicMolecule.diaIDs[fromAtom],
          paths: []
        };
        if (fromAtomicNo && molecule.getAtomicNo(fromAtom) !== fromAtomicNo) {
          continue;
        }
        for (let pathLength = minPathLength; pathLength <= maxPathLength; pathLength++) {
          const pathOfSpecificLength = atomsPaths[fromAtom][pathLength];
          for (const path of pathOfSpecificLength) {
            const toAtom = path.path.at(-1);
            if (toAtomicNo && molecule.getAtomicNo(toAtom) !== toAtomicNo) {
              continue;
            }
            results[fromAtom].paths.push({
              toDiaID: topicMolecule.diaIDs[toAtom],
              pathLength: path.pathLength,
              path: path.path,
              hoses: getHoseCodesForAtomsAsStrings(topicMolecule.moleculeWithH, {
                ...options,
                rootAtoms: path.path,
                tagAtoms: [fromAtom, toAtom]
              })
            });
          }
        }
      }
      return results;
    }

    function getDiaIDsAndInfo(diaMol, canonizedDiaIDs) {
      const newDiaIDs = [];
      const molecule = diaMol.moleculeWithH;
      const counts = {};
      for (const diaID of canonizedDiaIDs) {
        if (!counts[diaID]) {
          counts[diaID] = 0;
        }
        counts[diaID]++;
      }
      for (let i = 0; i < canonizedDiaIDs.length; i++) {
        const diaID = canonizedDiaIDs[diaMol.finalRanks[i]];
        if (!diaID) {
          throw new Error(`Unexpected missing canonized diaID for atom ${i}`);
        }
        const count = counts[diaID];
        if (!count) {
          throw new Error(`Unexpected missing count for diaID ${diaID}`);
        }
        const newDiaID = {
          idCode: diaID,
          attachedHydrogensIDCodes: [],
          attachedHydrogens: [],
          nbAttachedHydrogens: 0,
          atomLabel: molecule.getAtomLabel(i),
          nbEquivalentAtoms: count,
          heavyAtom: undefined,
          atomMapNo: molecule.getAtomMapNo(i)
        };
        if (molecule.getAtomicNo(i) === 1) {
          const atom = molecule.getConnAtom(i, 0);
          newDiaID.heavyAtom = canonizedDiaIDs[diaMol.finalRanks[atom]];
        }
        for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
          const atom = molecule.getConnAtom(i, j);
          if (molecule.getAtomicNo(atom) === 1) {
            newDiaID.nbAttachedHydrogens++;
            newDiaID.attachedHydrogens.push(atom);
            const hydrogenDiaID = canonizedDiaIDs[diaMol.finalRanks[atom]];
            if (!hydrogenDiaID) {
              throw new Error(`Unexpected missing canonized diaID for atom ${atom}`);
            }
            if (!newDiaID.attachedHydrogensIDCodes.includes(hydrogenDiaID)) {
              newDiaID.attachedHydrogensIDCodes.push(hydrogenDiaID);
            }
          }
        }
        newDiaIDs.push(newDiaID);
      }
      return newDiaIDs;
    }

    /**
     * Get a unique atomic number for a X
     * @param xMolecule
     * @returns
     */
    function getHeterotopicSymmetryRanks(xMolecule) {
      xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      const symmetryRanks = [];
      for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
        symmetryRanks.push(xMolecule.getSymmetryRank(i));
      }
      return symmetryRanks;
    }
    function getFinalRanks(xMolecule) {
      xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
      return xMolecule.getFinalRanks(0).map(rank => rank - 1);
    }

    /**
     * Returns the atoms that are chiral or pseudo chiral.
     * There could be some issues if the original molecule lacks chiral bonds.
     * The function will add them and this could lead to some issues in the case of pseudochiral atoms.
     * @param {import('openchemlib').Molecule} molecule
     * @returns {number[]}
     */
    function getChiralOrHeterotopicCarbons(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      const xAtomicNumber = getXAtomicNumber(molecule);
      const internalMolecule = molecule.getCompactCopy();
      // hydrogens may be diastereotopic, we need to add them
      internalMolecule.addImplicitHydrogens();
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (internalMolecule.getAtomicNo(i) === 1) {
          internalMolecule.setAtomicNo(i, xAtomicNumber);
        }
      }
      addPossibleChiralBonds(internalMolecule);
      internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const atoms = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (internalMolecule.getAtomicNo(i) === xAtomicNumber) {
          continue;
        }
        if (molecule.getAtomicNo(i) !== internalMolecule.getAtomicNo(i)) {
          throw new Error('getChiralOrHeterotopicCarbons: mismatching atomic numbers');
        }
        if (internalMolecule.getAtomicNo(i) !== 6) {
          continue;
        }
        const neighbourSymmetries = getNeighbourSymmetries(internalMolecule, i);
        if (neighbourSymmetries.length === 4) {
          atoms.push(i);
        }
      }
      return atoms;
    }
    function addPossibleChiralBonds(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      for (let i = 0; i < molecule.getAtoms(); i++) {
        if (molecule.getAtomicNo(i) !== 6) continue;
        if (molecule.getStereoBond(i) >= 0) continue;
        const neighbourSymmetries = getNeighbourSymmetries(molecule, i);
        if (neighbourSymmetries.length <= 2) continue;
        const stereoBond = molecule.getAtomPreferredStereoBond(i);
        if (stereoBond !== -1) {
          molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
          if (molecule.getBondAtom(1, stereoBond) === i) {
            const connAtom = molecule.getBondAtom(0, stereoBond);
            molecule.setBondAtom(0, stereoBond, i);
            molecule.setBondAtom(1, stereoBond, connAtom);
          }
          // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0);
        }
      }
    }
    function getNeighbourSymmetries(molecule, iAtom) {
      const neighbourSymmetries = [];
      for (let j = 0; j < molecule.getAllConnAtoms(iAtom); j++) {
        const connAtom = molecule.getConnAtom(iAtom, j);
        const symmetryRank = molecule.getSymmetryRank(connAtom);
        if (!neighbourSymmetries.includes(symmetryRank)) {
          neighbourSymmetries.push(molecule.getSymmetryRank(connAtom));
        }
      }
      return neighbourSymmetries;
    }

    /**
     * This function will add missing chiral bonds on carbons ensure that all enantiotopic
     * or diastereotopic atoms can be identified uniquely
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {number} [options.esrType=Molecule.cESRTypeAnd]
     * @param {boolean} [options.atLeastThreeAtoms=true] - if true, only carbons with at least three atoms will be considered
     */
    function ensureHeterotopicChiralBonds(molecule, options = {}) {
      const {
        Molecule
      } = molecule.getOCL();
      const {
        esrType = Molecule.cESRTypeAnd,
        atLeastThreeAtoms = true
      } = options;
      molecule.ensureHelperArrays(Molecule.cHelperBitNeighbours);
      const heterotopicCarbons = getChiralOrHeterotopicCarbons(molecule);
      for (const i of heterotopicCarbons) {
        if (atLeastThreeAtoms && molecule.getAllConnAtoms(i) < 3) continue;
        if (molecule.getStereoBond(i) === -1) {
          const stereoBond = molecule.getAtomPreferredStereoBond(i);
          if (stereoBond !== -1) {
            molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
            if (molecule.getBondAtom(1, stereoBond) === i) {
              const connAtom = molecule.getBondAtom(0, stereoBond);
              molecule.setBondAtom(0, stereoBond, i);
              molecule.setBondAtom(1, stereoBond, connAtom);
            }
            // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
            molecule.setAtomESR(i, esrType, 0);
          }
        }
      }
    }

    /**
     * Expand all the implicit hydrogens and ensure chiral bonds on heterotopic bonds
     * @param molecule
     * @param options
     * @returns
     */
    function getMoleculeWithH(molecule, options) {
      const {
        logger,
        maxNbAtoms
      } = options;
      const moleculeWithH = molecule.getCompactCopy();
      moleculeWithH.addImplicitHydrogens();
      if (moleculeWithH.getAllAtoms() > maxNbAtoms) {
        logger.warn(`too many atoms to evaluate heterotopic chiral bonds: ${moleculeWithH.getAllAtoms()} > ${maxNbAtoms}`);
      } else {
        ensureHeterotopicChiralBonds(moleculeWithH);
      }
      return moleculeWithH;
    }

    /**
     * In order to be able to give a unique ID to all the atoms we are replacing the H by X
     * @param moleculeWithH
     * @returns
     */
    function getXMolecule(moleculeWithH) {
      const xAtomNumber = getXAtomicNumber(moleculeWithH);
      const xMolecule = moleculeWithH.getCompactCopy();
      for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (xMolecule.getAtomicNo(i) === 1) {
          xMolecule.setAtomicNo(i, xAtomNumber);
        }
      }
      return xMolecule;
    }

    /**
     * This class deals with topicity information and hose codes
     * It is optimized to avoid recalculation of the same information
     */
    class TopicMolecule {
      originalMolecule;
      molecule;
      idCode;
      options;
      cache;
      constructor(molecule, options = {}) {
        this.originalMolecule = molecule;
        this.options = {
          maxPathLength: 5,
          maxNbAtoms: 250,
          logger: console,
          ...options
        };
        this.idCode = molecule.getIDCode();
        this.molecule = this.originalMolecule.getCompactCopy();
        this.molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
        this.molecule.ensureHelperArrays(this.molecule.getOCL().Molecule.cHelperNeighbours);
        this.cache = {};
      }
      /**
       * This method ensures that all the atoms have a mapNo corresponding to the atom number.
       * It will enforce mapNo in molecule and moleculeWithH
       * We start numbering the atoms at 1
       */
      setAtomNoInMapNo() {
        const molecules = [this.molecule, this.moleculeWithH];
        for (const molecule of molecules) {
          for (let i = 0; i < molecule.getAllAtoms(); i++) {
            molecule.setAtomMapNo(i, i + 1, false);
          }
        }
      }
      /**
       * This method ensures that all the atoms have a mapNo in the molecule (and not the moleculeWithH! )
       */
      ensureMapNo() {
        const existingMapNo = {};
        for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
          const mapNo = this.molecule.getAtomMapNo(i);
          if (mapNo) {
            if (existingMapNo[mapNo]) {
              throw new Error('The molecule contains several atoms with the same mapNo');
            }
            existingMapNo[mapNo] = true;
          }
        }
        let nextMapNo = 1;
        for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
          const mapNo = this.molecule.getAtomMapNo(i);
          if (!mapNo) {
            while (existingMapNo[nextMapNo]) {
              nextMapNo++;
            }
            existingMapNo[nextMapNo] = true;
            this.molecule.setAtomMapNo(i, nextMapNo, false);
          }
        }
      }
      /**
       * For each atom we will return an array of objects that contains the
       * different possible path as well as the canonic hose codes
       * @param options
       * @returns
       */
      getHoseCodesForPath(options = {}) {
        return getCanonizedHoseCodesForPath(this, options);
      }
      /**
       * Return one fragment for a specific sphere size and specific root atoms
       * @param rootAtoms
       * @param options
       * @returns
       */
      getHoseFragment(rootAtoms, options = {}) {
        const {
          sphereSize = 2,
          tagAtoms = rootAtoms,
          tagAtomFct
        } = options;
        const fragments = getHoseCodesForAtomsAsFragments(this.moleculeWithH, {
          rootAtoms,
          minSphereSize: sphereSize,
          maxSphereSize: sphereSize,
          tagAtoms,
          tagAtomFct
        });
        return fragments[0];
      }
      getAtomPathsFrom(atom, options = {}) {
        const {
          minPathLength = 1,
          maxPathLength = this.options.maxPathLength,
          toAtomicNo
        } = options;
        if (maxPathLength > this.options.maxPathLength) {
          throw new Error('The maxPathLength is too long, you should increase the maxPathLength when instantiating the TopicMolecule');
        }
        const atomPaths = this.atomsPaths[atom];
        if (!atomPaths) {
          throw new Error('Unexpected missing atom path');
        }
        const paths = [];
        for (let i = minPathLength; i <= maxPathLength; i++) {
          const atomPathValue = atomPaths[i];
          if (!atomPathValue) {
            throw new Error(`Unexpected missing atom path at index ${i}`);
          }
          for (const atomPath of atomPathValue) {
            if (!toAtomicNo || this.moleculeWithH.getAtomicNo(atomPath.path.at(-1)) === toAtomicNo) {
              paths.push(atomPath.path);
            }
          }
        }
        return paths;
      }
      getAtomPaths(atom1, atom2, options = {}) {
        const {
          pathLength
        } = options;
        if (pathLength !== undefined && pathLength > this.options.maxPathLength) {
          throw new Error('The distance is too long, you should increase the maxPathLength when instantiating the TopicMolecule');
        }
        const atomPaths = this.atomsPaths[atom1];
        if (!atomPaths) {
          throw new Error('Unexpected missing atom path');
        }
        const minDistance = pathLength || 0;
        const maxDistance = pathLength || this.options.maxPathLength;
        const paths = [];
        for (let i = minDistance; i <= maxDistance; i++) {
          const atomPathValue = atomPaths[i];
          if (!atomPathValue) {
            throw new Error(`Unexpected missing atom path at index ${i}`);
          }
          for (const atomPath of atomPathValue) {
            if (atomPath.path.at(-1) === atom2) {
              paths.push(atomPath.path);
            }
          }
        }
        return paths;
      }
      get atomsPaths() {
        if (this.cache.atomsPaths) return this.cache.atomsPaths;
        this.cache.atomsPaths = getAllAtomsPaths(this.moleculeWithH, {
          maxPathLength: this.options.maxPathLength
        });
        return this.cache.atomsPaths;
      }
      toMolfile(options = {}) {
        const {
          version = 2
        } = options;
        if (version === 2) {
          return this.molecule.toMolfile();
        }
        return this.molecule.toMolfileV3();
      }
      getMolecule() {
        return this.molecule;
      }
      /**
       * Returns a new TopicMolecule but will copy precalculated information
       * if possible (same idCode). This is very practical when expanding hydrogens
       * for example.
       * @param molecule
       * @returns
       */
      fromMolecule(molecule) {
        const idCode = molecule.getIDCode();
        if (idCode !== this.idCode) {
          // no way for optimisation
          return new TopicMolecule(molecule);
        }
        const topicMolecule = new TopicMolecule(molecule);
        topicMolecule.cache = {
          canonizedDiaIDs: this.cache.canonizedDiaIDs,
          canonizedHoseCodes: this.cache.canonizedHoseCodes
        };
        return topicMolecule;
      }
      /**
       * Returns a molecule with all the hydrogens added. The order is NOT canonized
       */
      get moleculeWithH() {
        if (this.cache.moleculeWithH) return this.cache.moleculeWithH;
        this.cache.moleculeWithH = getMoleculeWithH(this.molecule, {
          maxNbAtoms: this.options.maxNbAtoms,
          logger: this.options.logger
        });
        return this.cache.moleculeWithH;
      }
      get xMolecule() {
        if (this.cache.xMolecule) return this.cache.xMolecule;
        this.cache.xMolecule = getXMolecule(this.moleculeWithH);
        return this.cache.xMolecule;
      }
      /**
       * This is related to the current moleculeWithH. The order is NOT canonized
       */
      get diaIDs() {
        if (this.cache.diaIDs) return this.cache.diaIDs;
        const diaIDs = [];
        if (this.moleculeWithH.getAllAtoms() > this.options.maxNbAtoms) {
          this.options.logger.warn(`too many atoms to evaluate heterotopicity: ${this.moleculeWithH.getAllAtoms()} > ${this.options.maxNbAtoms}`);
        } else {
          for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
            diaIDs.push(this.canonizedDiaIDs[this.finalRanks[i]]);
          }
        }
        this.cache.diaIDs = diaIDs;
        return diaIDs;
      }
      /**
       * We return the atomIDs corresponding to the specified diaID as well has the attached hydrogens or heavy atoms
       * @param diaID
       * @returns
       */
      getDiaIDsObject() {
        return groupDiastereotopicAtomIDsAsObject(this.diaIDs, this.molecule, this.moleculeWithH);
      }
      /**
       * This is related to the current moleculeWithH. The order is NOT canonized
       */
      get hoseCodes() {
        if (this.cache.hoseCodes) return this.cache.hoseCodes;
        const hoseCodes = [];
        for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
          hoseCodes.push(this.canonizedHoseCodes[this.finalRanks[i]]);
        }
        this.cache.hoseCodes = hoseCodes;
        return hoseCodes;
      }
      get canonizedDiaIDs() {
        if (this.cache.canonizedDiaIDs) return this.cache.canonizedDiaIDs;
        this.cache.canonizedDiaIDs = getCanonizedDiaIDs(this, {
          maxNbAtoms: this.options.maxNbAtoms,
          logger: this.options.logger
        });
        return this.cache.canonizedDiaIDs;
      }
      get canonizedHoseCodes() {
        if (this.cache.canonizedHoseCodes) {
          return this.cache.canonizedHoseCodes;
        }
        this.cache.canonizedHoseCodes = getCanonizedHoseCodes(this);
        return this.cache.canonizedHoseCodes;
      }
      /**
       * Returns the distance matrix for the current moleculeWithH
       */
      get distanceMatrix() {
        return getConnectivityMatrix(this.moleculeWithH, {
          pathLength: true
        });
      }
      get diaIDsAndInfo() {
        if (this.cache.diaIDsAndInfo) {
          return this.cache.diaIDsAndInfo;
        }
        this.cache.diaIDsAndInfo = getDiaIDsAndInfo(this, this.canonizedDiaIDs);
        return this.cache.diaIDsAndInfo;
      }
      /**
       * Returns symmetryRanks for all the atoms including hydrogens. Those ranks
       * deals with topicity and is related to the current moleculeWithH.
       * In order to calculate the ranks we replace all the
       * hydrogens with a X atom.
       */
      get heterotopicSymmetryRanks() {
        if (this.cache.heterotopicSymmetryRanks) {
          return this.cache.heterotopicSymmetryRanks;
        }
        this.cache.heterotopicSymmetryRanks = getHeterotopicSymmetryRanks(this.xMolecule);
        return [...this.cache.heterotopicSymmetryRanks];
      }
      /**
       * Returns finalRanks for all the atoms including hydrogens. Those ranks
       * deals with topicity and is related to the current moleculeWithH.
       * All the atoms have a unique identifier.j
       * In order to calculate the ranks we replace all the
       * hydrogens with a X atom.
       */
      get finalRanks() {
        if (this.cache.finalRanks) return this.cache.finalRanks;
        this.cache.finalRanks = getFinalRanks(this.xMolecule);
        return this.cache.finalRanks;
      }
      toMolfileWithH(options = {}) {
        const {
          version = 2
        } = options;
        if (version === 2) {
          return this.moleculeWithH.toMolfile();
        }
        return this.moleculeWithH.toMolfileV3();
      }
      toMolfileWithoutH(options = {}) {
        const molecule = this.molecule.getCompactCopy();
        molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
        const atomsToDelete = [];
        for (let atomID = 0; atomID < molecule.getAtoms(); atomID++) {
          if (molecule.getExplicitHydrogens(atomID) > 0) {
            for (let i = 0; i < molecule.getAllConnAtoms(atomID); i++) {
              const connectedAtom = molecule.getConnAtom(atomID, i);
              if (molecule.getAtomicNo(connectedAtom) === 1) {
                atomsToDelete.push(connectedAtom);
              }
            }
          }
        }
        molecule.deleteAtoms(atomsToDelete);
        const {
          version = 2
        } = options;
        if (version === 2) {
          return molecule.toMolfile();
        }
        return molecule.toMolfileV3();
      }
      /**
       * Returns an array of objects containing the oclID and the corresponding hydrogens and atoms
       * for the specified atomLabel (if any)
       * This always applied to the molecule with expanded hydrogens and chirality
       * @param options
       * @returns
       */
      getGroupedDiastereotopicAtomIDs(options = {}) {
        if (!this.diaIDs) return undefined;
        return groupDiastereotopicAtomIDs$1(this.diaIDs, this.moleculeWithH, options);
      }
      /**
       * This method returns a mapping between the diaIDs of the current molecule.
       * It expects that the initial molfile and the final molfile contains atomMapNo
       * in order to track which atom becomes which one.
       * @param molecule
       */
      getDiaIDsMapping(molecule) {
        const topicMolecule = new TopicMolecule(molecule);
        const originalDiaIDs = this.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
        const destinationDiaIDs = topicMolecule.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
        const mapping = {};
        // we first check all the atoms present in the molfile
        for (const destinationDiaID of destinationDiaIDs) {
          const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
          const newIDCode = destinationDiaID.idCode;
          const oldIDCode = originalDiaID.idCode;
          if (oldIDCode in mapping) {
            if (mapping[oldIDCode] !== newIDCode) {
              mapping[oldIDCode] = undefined;
            }
          } else {
            mapping[oldIDCode] = newIDCode;
          }
        }
        // we now check all the attached hydrogens that are not defined in the molfile and were not yet mapped
        for (const destinationDiaID of destinationDiaIDs) {
          const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
          for (let i = 0; i < originalDiaID.attachedHydrogensIDCodes.length; i++) {
            const oldHydrogenIDCode = originalDiaID.attachedHydrogensIDCodes.at(i);
            if (mapping[oldHydrogenIDCode]) continue;
            const newHydrogenIDCode = destinationDiaID.attachedHydrogensIDCodes[i];
            if (oldHydrogenIDCode && newHydrogenIDCode) {
              if (oldHydrogenIDCode in mapping) {
                if (mapping[oldHydrogenIDCode] !== newHydrogenIDCode) {
                  mapping[oldHydrogenIDCode] = undefined;
                }
              } else {
                mapping[oldHydrogenIDCode] = newHydrogenIDCode;
              }
            }
          }
        }
        return mapping;
      }
    }
    function groupDiastereotopicAtomIDs$1(diaIDs, molecule, options = {}) {
      const diaIDsObject = groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, molecule, options);
      return Object.values(diaIDsObject);
    }
    function groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, moleculeWithH, options = {}) {
      const {
        atomLabel
      } = options;
      const diaIDsObject = {};
      for (let i = 0; i < diaIDs.length; i++) {
        if (!atomLabel || moleculeWithH.getAtomLabel(i) === atomLabel) {
          const diaID = diaIDs[i];
          if (!diaIDsObject[diaID]) {
            diaIDsObject[diaID] = {
              counter: 0,
              oclID: diaID,
              atomLabel: moleculeWithH.getAtomLabel(i),
              atoms: [],
              existingAtoms: [],
              heavyAtoms: [],
              attachedHydrogens: [],
              customLabels: [],
              attachedHydrogensCustomLabels: [],
              heavyAtomsCustomLabels: []
            };
          }
          const customLabel = moleculeWithH.getAtomCustomLabel(i)?.replace(/^\]/, '');
          if (customLabel && !diaIDsObject[diaID].customLabels.includes(customLabel)) {
            diaIDsObject[diaID].customLabels.push(customLabel);
          }
          if (moleculeWithH.getAtomicNo(i) === 1) {
            const connected = moleculeWithH.getConnAtom(i, 0);
            if (!diaIDsObject[diaID].heavyAtoms.includes(connected)) {
              diaIDsObject[diaID].heavyAtoms.push(connected);
            }
            const heavyAtomCustomLabel = molecule.getAtomCustomLabel(connected)?.replace(/^\]/, '');
            if (heavyAtomCustomLabel && !diaIDsObject[diaID].heavyAtomsCustomLabels.includes(heavyAtomCustomLabel)) {
              diaIDsObject[diaID].heavyAtomsCustomLabels.push(heavyAtomCustomLabel);
            }
            if (molecule.getAtomicNo(i)) {
              diaIDsObject[diaID].existingAtoms.push(i);
            } else if (!diaIDsObject[diaID].existingAtoms.includes(connected)) {
              diaIDsObject[diaID].existingAtoms.push(connected);
            }
          } else {
            for (let j = 0; j < moleculeWithH.getAllConnAtoms(i); j++) {
              const connected = moleculeWithH.getConnAtom(i, j);
              if (moleculeWithH.getAtomicNo(connected) === 1) {
                const attachedHydrogenCustomLabel = molecule.getAtomCustomLabel(connected)?.replace(/^\]/, '');
                if (attachedHydrogenCustomLabel && !diaIDsObject[diaID].attachedHydrogensCustomLabels.includes(attachedHydrogenCustomLabel)) {
                  diaIDsObject[diaID].attachedHydrogensCustomLabels.push(attachedHydrogenCustomLabel);
                }
                diaIDsObject[diaID].attachedHydrogens.push(connected);
              }
            }
            diaIDsObject[diaID].existingAtoms.push(i);
          }
          diaIDsObject[diaID].counter++;
          diaIDsObject[diaID].atoms.push(i);
        }
      }
      for (const diaIDObject of Object.values(diaIDsObject)) {
        diaIDObject.existingAtoms.sort((a, b) => a - b);
        diaIDObject.attachedHydrogens.sort((a, b) => a - b);
      }
      return diaIDsObject;
    }

    /**
     * Returns an array of symmetry ranks.
     * @param {import('openchemlib').Molecule} molecule - An instance of a molecule
     * @returns
     */
    function getSymmetryRanks(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      const xAtomNumber = getXAtomicNumber(molecule);
      // most of the molecules have some symmetry
      const internalMolecule = molecule.getCompactCopy();
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (internalMolecule.getAtomicNo(i) === 1) {
          internalMolecule.setAtomicNo(i, xAtomNumber);
        }
      }
      internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const symmetryRanks = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        symmetryRanks.push(internalMolecule.getSymmetryRank(i));
      }
      return symmetryRanks;
    }

    /**
     * Returns an array of diastereotopic ID (as idCode)
     * @param {import('openchemlib').Molecule} molecule
     */
    function getDiastereotopicAtomIDs(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      ensureHeterotopicChiralBonds(molecule);
      const symmetryRanks = getSymmetryRanks(molecule);
      const numberAtoms = molecule.getAllAtoms();
      const ids = [];
      const cache = {};
      for (let iAtom = 0; iAtom < numberAtoms; iAtom++) {
        const rank = symmetryRanks[iAtom];
        if (rank && cache[rank]) {
          ids[iAtom] = cache[rank];
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(molecule);
        tagAtom(tempMolecule, iAtom);
        makeRacemic(tempMolecule);
        // We need to ensure the helper array in order to get correctly the result of racemisation
        ids[iAtom] = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        cache[rank] = ids[iAtom];
      }
      return ids;
    }

    /**
     *
     * @param {import('openchemlib').Molecule} molecule
     */
    function getDiastereotopicAtomIDsAndH(molecule) {
      const OCL = molecule.getOCL();
      molecule = molecule.getCompactCopy();
      molecule.addImplicitHydrogens();
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      const diaIDs = getDiastereotopicAtomIDs(molecule);
      const newDiaIDs = [];
      for (let i = 0; i < diaIDs.length; i++) {
        const diaID = diaIDs[i];
        const newDiaID = {
          oclID: diaID,
          hydrogenOCLIDs: [],
          nbHydrogens: 0
        };
        if (molecule.getAtomicNo(i) === 1) {
          const atom = molecule.getConnAtom(i, 0);
          newDiaID.heavyAtom = diaIDs[atom];
        }
        for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
          const atom = molecule.getConnAtom(i, j);
          if (molecule.getAtomicNo(atom) === 1) {
            newDiaID.nbHydrogens++;
            if (!newDiaID.hydrogenOCLIDs.includes(diaIDs[atom])) {
              newDiaID.hydrogenOCLIDs.push(diaIDs[atom]);
            }
          }
        }
        newDiaIDs.push(newDiaID);
      }
      return newDiaIDs;
    }

    function groupDiastereotopicAtomIDs(diaIDs, molecule, options = {}) {
      const {
        atomLabel
      } = options;
      const diaIDsObject = {};
      for (let i = 0; i < diaIDs.length; i++) {
        if (!atomLabel || molecule.getAtomLabel(i) === atomLabel) {
          const diaID = diaIDs[i];
          if (!diaIDsObject[diaID]) {
            diaIDsObject[diaID] = {
              counter: 0,
              atoms: [],
              oclID: diaID,
              atomLabel: molecule.getAtomLabel(i)
            };
          }
          diaIDsObject[diaID].counter++;
          diaIDsObject[diaID].atoms.push(i);
        }
      }
      return Object.keys(diaIDsObject).map(key => diaIDsObject[key]);
    }

    /**
     * This function groups the diasterotopic atomIds of the molecule based on equivalence of atoms. The output object contains
     * a set of chemically equivalent atoms(element.atoms) and the groups of magnetically equivalent atoms (element.magneticGroups)
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {string} [options.atomLabel] - Select atoms of the given atomLabel. By default it returns all the explicit atoms in the molecule
     * @returns {Array}
     */
    function getGroupedDiastereotopicAtomIDs(molecule, options = {}) {
      const diaIDs = getDiastereotopicAtomIDs(molecule);
      return groupDiastereotopicAtomIDs(diaIDs, molecule, options);
    }

    /**
     * Parse a molfile and returns an object containing the molecule, the map and the diaIDs
     * The map allows to reload properties assigned to the atom molfile
     * Please take care than numbering of atoms starts at 0 !
     * @param {typeof import('openchemlib')} OCL - openchemlib library
     * @param {string} molfile
     * @returns
     */
    function getDiastereotopicAtomIDsFromMolfile(OCL, molfile) {
      const {
        map,
        molecule
      } = OCL.Molecule.fromMolfileWithAtomMap(molfile);
      const diaIDsArray = getDiastereotopicAtomIDsAndH(molecule);
      const diaIDs = {};
      for (let i = 0; i < map.length; i++) {
        diaIDs[map[i]] = {
          source: map[i],
          destination: i,
          ...diaIDsArray[i]
        };
      }
      return {
        map: diaIDs,
        molecule,
        diaIDs: diaIDsArray
      };
    }

    /**
     * Returns a SVG
     * @param {*} molecule
     * @param {*} [options={}]
     */
    function toDiastereotopicSVG(molecule, options = {}) {
      const {
        width = 300,
        height = 200,
        prefix = 'ocl',
        heavyAtomHydrogen = false
      } = options;
      let svg = options.svg;
      let diaIDs = [];
      const hydrogenInfo = {};
      for (const line of getDiastereotopicAtomIDsAndH(molecule)) {
        hydrogenInfo[line.oclID] = line;
      }
      if (heavyAtomHydrogen) {
        for (let i = 0; i < molecule.getAtoms(); i++) {
          diaIDs.push([]);
        }
        const groupedDiaIDs = molecule.getGroupedDiastereotopicAtomIDs();
        for (const diaID of groupedDiaIDs) {
          if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
            for (const atom of diaID.atoms) {
              for (const id of hydrogenInfo[diaID.oclID].hydrogenOCLIDs) {
                if (!diaIDs[atom * 1].includes(id)) diaIDs[atom].push(id);
              }
            }
          }
        }
      } else {
        diaIDs = molecule.getDiastereotopicAtomIDs().map(a => [a]);
      }
      if (!svg) svg = molecule.toSVG(width, height, prefix);
      svg = svg.replaceAll(/Atom:\d+"/g, value => {
        const atom = value.replaceAll(/\D/g, '');
        return `${value} data-diaid="${diaIDs[atom].join(',')}"`;
      });
      return svg;
    }

    /**
     * Returns the hose codes for all atoms in the molecule
     * @param {*} molecule
     * @param {object} [options={}]
     * @param {string[]} [options.atomLabels]
     * @param {number} [options.minSphereSize=0]
     * @param {number} [options.maxSphereSize=4]
     * @returns
     */
    function getHoseCodes(molecule, options = {}) {
      const {
        atomLabels,
        minSphereSize,
        maxSphereSize
      } = options;
      const {
        Molecule
      } = molecule.getOCL();
      const atomicNumbers = atomLabels?.map(label => Molecule.getAtomicNoFromLabel(label));
      const internalMolecule = getCompactCopyWithoutCustomLabels(molecule);
      internalMolecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(internalMolecule);
      const hoses = [];
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        if (atomicNumbers && !atomicNumbers.includes(internalMolecule.getAtomicNo(i))) {
          hoses.push(undefined);
        } else {
          const tempMolecule = internalMolecule.getCompactCopy();
          tagAtom(tempMolecule, i);
          hoses.push(getHoseCodesForAtomsAsStrings(tempMolecule, {
            minSphereSize,
            maxSphereSize
          }));
        }
      }
      return hoses;
    }

    /**
     * Returns an object containing a molfile, molfile with hydrogens, hoses codes and optionally the diaIDs
     * and the diaIDs
     * The map allows to reload properties assigned to the atom molfile
     * Please take care than numbering of atoms starts at 0 !
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {string[]} [options.atomLabels]
     * @param {boolean} [options.calculateDiastereotopicIDs=true]
     * @param {number} [options.minSphereSize=0]
     * @param {number} [options.maxSphereSize=4]
     * @returns
     */
    function getHoseCodesAndInfo(molecule, options = {}) {
      const {
        minSphereSize,
        maxSphereSize,
        calculateDiastereotopicIDs = true
      } = options;
      const {
        Molecule
      } = molecule.getOCL();
      molecule = molecule.getCompactCopy();
      // this will force reordering of the hydrogens to the end, just to have the same order as in the molfile
      molecule.ensureHelperArrays(Molecule.cHelperNeighbours);
      const newMolfile = molecule.toMolfile();
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const newMolfileWithH = molecule.toMolfile();
      const symmetryRanks = getSymmetryRanks(molecule);
      const cache = {};
      const hoses = [];
      const diaIDs = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        const rank = symmetryRanks[i];
        if (rank && cache[rank]) {
          diaIDs.push(cache[rank].diaID);
          hoses.push(cache[rank].hose);
          continue;
        }
        const tempMolecule = getCompactCopyWithoutCustomLabels(molecule);
        tagAtom(tempMolecule, i);
        let diaID;
        if (calculateDiastereotopicIDs) {
          makeRacemic(tempMolecule);
          diaID = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
          diaIDs.push(diaID);
        }
        const hose = getHoseCodesForAtomsAsStrings(tempMolecule, {
          minSphereSize,
          maxSphereSize
        });
        hoses.push(hose);
        cache[rank] = {
          diaID,
          hose
        };
      }
      const distanceMatrix = getConnectivityMatrix(molecule, {
        pathLength: true
      });
      return {
        molfile: newMolfile,
        molfileWithH: newMolfileWithH,
        hoses,
        diaIDs: calculateDiastereotopicIDs ? diaIDs : undefined,
        moleculeWithHydrogens: molecule,
        distanceMatrix
      };
    }

    const FULL_HOSE_CODE = 1;
    const HOSE_CODE_CUT_C_SP3_SP3 = 2;
    /**
     * Returns the hose code for specific atom numbers
     * @param {import('openchemlib').Molecule} originalMolecule - The OCL molecule to be fragmented
     * @param {number[]} rootAtoms
     * @param {object} [options={}]
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtoms(originalMolecule, rootAtoms = [], options = {}) {
      const {
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE
      } = options;
      const molecule = getCompactCopyWithoutCustomLabels(originalMolecule);
      // those 2 lines should be done only once
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      const allowedCustomLabels = [];
      for (const rootAtom of rootAtoms) {
        allowedCustomLabels.push(tagAtom(molecule, rootAtom));
      }
      return getHoseCodesForAtomsAsStrings(molecule, {
        minSphereSize,
        maxSphereSize,
        allowedCustomLabels,
        kind
      });
    }

    /**
     * Returns the hose code for a specific atom number
     * @param {import('openchemlib').Molecule} originalMolecule
     * @param {number} rootAtom
     * @param {object} [options={}]
     * @param {boolean} [options.isTagged] - Specify is the atom is already tagged
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtom(originalMolecule, rootAtom, options = {}) {
      return getHoseCodesForAtoms(originalMolecule, [rootAtom], options);
    }

    /**
     * Returns an array of strings (idCodes) specified molecule. Each string corresponds to a
     * hose code. By default it will calculate the hose codes for sphere 0 to 4 and will reuse
     * the existing tagged atoms.
     * This method ensure implicity hydrogens and possible missing chiral bonds.
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to process.
     * @param {object} options - Options for generating hose codes.
     * @param {string[]} [options.allowedCustomLabels] - Array of the custom labels that are considered as root atoms. By default all atoms having a customLabel
     * @param {number} [options.minSphereSize=0] - Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] - Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] - Kind of hose code, default usual sphere
     *  @param {number[]} [options.rootAtoms=[]] - Array of atom from which we should start to create the HOSE. By default we will used the taggedAtoms
     * @param {number[]} [options.tagAtoms=[]] - Array of atom indices to tag as root atoms
     * @param {Function} [options.tagAtomFct=tagAtom] - Function to tag an atom as root atom. By default it is defined internal
     * @returns {Array} - An array of hose code fragments.
     */
    function getHoseCodesFromDiastereotopicID(molecule, options = {}) {
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      return getHoseCodesForAtomsAsStrings(molecule, options);
    }

    let fragment$1;
    /**
     * Returns the hose code for a specific atom number
     * @param {import('openchemlib').Molecule} molecule
     * @param from
     * @param to
     * @param maxLength
     */
    function getHoseCodesForPath(molecule, from, to, maxLength) {
      const OCL = molecule.getOCL();
      const originalFrom = from;
      const originalTo = to;
      molecule = getCompactCopyWithoutCustomLabels(molecule);
      const originalAtoms = []; // path before renumbering
      molecule.getPath(originalAtoms, from, to, maxLength + 1);
      let torsion;
      if (originalAtoms.length === 4) {
        torsion = molecule.calculateTorsion(originalAtoms);
      }
      const tag1 = tagAtom(molecule, from);
      const tag2 = tagAtom(molecule, to);
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      from = -1;
      to = -1;
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (tag1 === tag2) {
          if (molecule.getAtomCustomLabel(i) === tag1) {
            if (from === -1) {
              from = i;
            } else {
              to = i;
            }
          }
        } else {
          if (tag1 === molecule.getAtomCustomLabel(i)) {
            from = i;
          }
          if (tag2 === molecule.getAtomCustomLabel(i)) {
            to = i;
          }
        }
      }
      if (!fragment$1) fragment$1 = new OCL.Molecule(0, 0);
      const atoms = [];
      molecule.getPath(atoms, from, to, maxLength + 1);
      let min = 0;
      let max = 0;
      const atomMask = new Array(molecule.getAllAtoms()).fill(false);
      const atomList = new Array(molecule.getAllAtoms()).fill(-1);
      const hoses = [];
      for (let sphere = 0; sphere <= 2; sphere++) {
        if (max === 0) {
          for (const atom of atoms) {
            atomMask[atom] = true;
            atomList[max++] = atom;
          }
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            const atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              const connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                atomMask[connAtom] = true;
                atomList[newMax++] = connAtom;
              }
            }
          }
          min = max;
          max = newMax;
        }
        const atomMap = [];
        molecule.copyMoleculeByAtoms(fragment$1, atomMask, true, atomMap);
        makeRacemic(fragment$1);
        const oclID = fragment$1.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
        hoses.push({
          sphere,
          oclID
        });
      }
      return {
        atoms: originalAtoms,
        from: originalFrom,
        to: originalTo,
        torsion,
        hoses,
        length: originalAtoms.length - 1
      };
    }

    /**
     * Create a polymer from a unit, alpha and omega groups
     * A unit must contain a R1 and a R2
     * An alpha end group must contain a R1
     * An omega end group must contain a R2
     * @param {import('openchemlib').Molecule} unit - an instance of OCL.Molecule
     * @param {object} options
     * @param {number} [options.count=10] - number of units
     * @param {boolean} [options.markMonomer=false] - mark the different units of the polymer in the atom map
     * @param {import('openchemlib').Molecule} [options.alpha] - alpha end group, default is an hydrogen
     * @param {import('openchemlib').Molecule} [options.gamma] - omega end group, default is an hydrogen
     */
    function createPolymer(unit, options = {}) {
      const {
        count = 10
      } = options;
      checkEntity(unit, 'unit');
      const {
        Molecule
      } = unit.getOCL();
      const {
        alpha,
        gamma
      } = getAlphaGamma(unit, options);
      checkEntity(alpha, 'alpha');
      checkEntity(gamma, 'gamma');
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      const polymer = alpha.getCompactCopy();
      polymer.addMolecule(getUnit(unit, 1, options));
      addBond(polymer, r1AtomicNo, r1AtomicNo);
      for (let i = 0; i < count - 1; i++) {
        polymer.addMolecule(getUnit(unit, i + 2, options));
        addBond(polymer, r2AtomicNo, r1AtomicNo);
      }
      polymer.addMolecule(gamma);
      addBond(polymer, r2AtomicNo, r2AtomicNo);
      polymer.ensureHelperArrays(Molecule.cHelperNeighbours);
      // encoding directly in atomNapNo didn't work out because it was removed when deleting atoms
      for (let i = 0; i < polymer.getAtoms(); i++) {
        polymer.setAtomMapNo(i, (polymer.getAtomCustomLabel(i) || '').replace(/monomer_/, ''));
        polymer.setAtomCustomLabel(i, '');
      }
      return polymer;
    }
    function getUnit(unit, index, options) {
      const {
        markMonomer = false
      } = options;
      if (markMonomer) {
        unit = unit.getCompactCopy();
        unit.ensureHelperArrays(unit.getOCL().Molecule.cHelperNeighbours);
        for (let j = 0; j < unit.getAtoms(); j++) {
          unit.setAtomCustomLabel(j, `monomer_${index}`);
        }
      }
      return unit;
    }
    function addBond(molecule, firstAtomicNo, secondAtomicNo) {
      molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
      let i, j;
      loop: for (i = 0; i < molecule.getAtoms(); i++) {
        if (molecule.getAtomicNo(i) === firstAtomicNo) {
          for (j = i + 1; j < molecule.getAtoms(); j++) {
            if (molecule.getAtomicNo(j) === secondAtomicNo) {
              molecule.addBond(molecule.getConnAtom(i, 0), molecule.getConnAtom(j, 0), 1);
              break loop;
            }
          }
        }
      }
      molecule.deleteAtoms([i, j]);
    }
    function checkEntity(unit, kind) {
      let nbR1 = 1;
      let nbR2 = 1;
      switch (kind) {
        case 'unit':
          break;
        case 'alpha':
          nbR2 = 0;
          break;
        case 'gamma':
          nbR1 = 0;
          break;
        default:
          throw new Error('Unknown kind');
      }
      if (!unit) {
        throw new Error('unit is required');
      }
      const {
        Molecule
      } = unit.getOCL();
      // unit must contain ONE R1 and ONE R2
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      let r1Count = 0;
      let r2Count = 0;
      for (let i = 0; i < unit.getAtoms(); i++) {
        if (unit.getAtomicNo(i) === r1AtomicNo) {
          r1Count++;
        }
        if (unit.getAtomicNo(i) === r2AtomicNo) {
          r2Count++;
        }
      }
      if (r1Count !== nbR1) {
        throw new Error(`${kind} must contain ${nbR1} R1`);
      }
      if (r2Count !== nbR2) {
        throw new Error(`${kind} must contain ${nbR2} R2`);
      }
    }
    function getAlphaGamma(unit, options) {
      let {
        alpha,
        gamma
      } = options;
      const {
        Molecule
      } = unit.getOCL();
      const {
        r1AtomicNo,
        r2AtomicNo
      } = getR1R2AtomicNo(Molecule);
      if (!alpha) {
        alpha = Molecule.fromSmiles('CC');
        alpha.setAtomicNo(0, r1AtomicNo);
        alpha.setAtomicNo(1, 1);
      }
      if (!gamma) {
        gamma = Molecule.fromSmiles('CC');
        gamma.setAtomicNo(0, r2AtomicNo);
        gamma.setAtomicNo(1, 1);
      }
      return {
        alpha,
        gamma
      };
    }
    function getR1R2AtomicNo(Molecule) {
      const r1AtomicNo = Molecule.getAtomicNoFromLabel('R1', Molecule.cPseudoAtomsRGroups);
      const r2AtomicNo = Molecule.getAtomicNoFromLabel('R2', Molecule.cPseudoAtomsRGroups);
      return {
        r1AtomicNo,
        r2AtomicNo
      };
    }

    var xr=class{copyMolecule(ft){let dt=ft.getIDCodeAndCoordinates();navigator.clipboard.writeText(`${dt.idCode} ${dt.coordinates}`);}pasteMolecule(){return null;}};var Tu="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",Ee=new Uint8Array(256);for(let d=0;d<Tu.length;d++)Ee[Tu.charCodeAt(d)]=d;function hu(d){let ft=d.length*.75,dt=d.length,jt,Gt=0,p1,Ot,at,t1;d.at(-1)==="="&&(ft--,d.at(-2)==="="&&ft--);let g1=new ArrayBuffer(ft),T1=new Uint8Array(g1);for(jt=0;jt<dt;jt+=4)p1=Ee[d.charCodeAt(jt)],Ot=Ee[d.charCodeAt(jt+1)],at=Ee[d.charCodeAt(jt+2)],t1=Ee[d.charCodeAt(jt+3)],T1[Gt++]=p1<<2|Ot>>4,T1[Gt++]=(Ot&15)<<4|at>>2,T1[Gt++]=(at&3)<<6|t1&63;return g1;}function Nr(d){return d.toString(16).padStart(2,"0");}var _A=globalThis.devicePixelRatio||1;var $n=class{constructor(ft){this.ctx=ft,this.ctx.textAlign="left",this.ctx.textBaseline="top",this.currentFontSize=12,this.currentFont="12px sans-serif",this.ctx.font=this.currentFont,this.currentColor="#000000",this.currentLineWidth=1,this.canvasCache=new Map();}clearRect(ft,dt,jt,Gt){this.ctx.clearRect(ft,dt,jt,Gt);}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}getSelectionBackgroundRGB(){return 12310268;}getLineWidth(){return this.currentLineWidth;}setRGB(ft){let dt=ft>>>16&255,jt=ft>>>8&255,Gt=ft>>>0&255;this.currentColor=`#${Nr(dt)}${Nr(jt)}${Nr(Gt)}`,this.ctx.fillStyle=this.currentColor,this.ctx.strokeStyle=this.currentColor;}setFont(ft,dt,jt){this.currentFontSize=ft,this.currentFont=`${dt?"bold":""} ${jt?"italic":""} ${ft}px sans-serif`,this.ctx.font=this.currentFont;}getFontSize(){return this.currentFontSize;}getBounds(ft){let dt=this.ctx.measureText(ft);return {x:dt.actualBoundingBoxLeft,y:dt.actualBoundingBoxAscent,width:dt.actualBoundingBoxRight,height:dt.actualBoundingBoxAscent};}drawString(ft,dt,jt){this.ctx.fillText(jt,ft,dt);}drawCenteredString(ft,dt,jt){this.ctx.textAlign="center",this.ctx.textBaseline="middle",this.ctx.fillText(jt,ft,dt),this.ctx.textAlign="left",this.ctx.textBaseline="top";}setLineWidth(ft){this.currentLineWidth=ft,this.ctx.lineWidth=ft;}fillRectangle(ft,dt,jt,Gt){this.ctx.fillRect(ft,dt,jt,Gt);}fillCircle(ft,dt,jt){let Gt=jt/2;this.ctx.beginPath(),this.ctx.arc(ft+Gt,dt+Gt,Gt,0,2*Math.PI),this.ctx.fill();}drawLine(ft,dt,jt,Gt){this.ctx.beginPath(),this.ctx.moveTo(ft,dt),this.ctx.lineTo(jt,Gt),this.ctx.stroke();}drawPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let dt=1;dt<ft.getSize();dt++)this.ctx.lineTo(ft.getX(dt),ft.getY(dt));this.ctx.stroke();}drawRectangle(ft,dt,jt,Gt){this.ctx.strokeRect(ft,dt,jt,Gt);}fillPolygon(ft){this.ctx.beginPath(),this.ctx.moveTo(ft.getX(0),ft.getY(0));for(let dt=1;dt<ft.getSize();dt++)this.ctx.lineTo(ft.getX(dt),ft.getY(dt));this.ctx.fill();}drawImage(ft,dt,jt,Gt,p1,Ot,at,t1,g1){if(arguments.length!==9)throw new Error(`drawImage call with ${arguments.length} arguments unimplemented`);let T1=this.canvasCache.get(ft);if(!T1){T1=document.createElement("canvas");let d1=ft.imageData;T1.width=d1.width,T1.height=d1.height;let o1=T1.getContext("2d");o1.globalAlpha=0,o1.putImageData(d1,0,0),this.canvasCache.set(ft,T1);}this.ctx.drawImage(T1,dt,jt,Gt,p1,Ot,at,t1,g1);}isDarkBackground(){return  false;}};var Gr=class{constructor(ft,dt){this.canvasElement=ft,this.changeListener=dt,this.drawContext=new $n(this.canvasElement.getContext("2d"));}getBackgroundRGB(){return 16777215;}getCanvasWidth(){return this.canvasElement.width;}getCanvasHeight(){return this.canvasElement.height;}getDrawContext(){return this.drawContext;}onChange(ft,dt){this.changeListener?.({what:ft,isUserEvent:dt});}getClipboardHandler(){return new xr();}};var f2=`
/* We can customize editor styles here. */
`,C8;function cu(){if(C8)return C8;let d=new CSSStyleSheet();return d.replaceSync(f2),C8=d,d;}function S8(d,ft,dt){let jt=-1;function Gt(Ot,at,t1=0){at.button>0||ft.fireMouseEvent(Ot,at.button+1,t1,Math.round(at.offsetX*_A),Math.round(at.offsetY*_A),at.shiftKey,at.ctrlKey,at.altKey,at.button===2);}d.addEventListener("pointerdown",Ot=>{jt===-1&&(jt=Ot.pointerId,Gt(dt.MOUSE_EVENT_PRESSED,Ot));});function p1(Ot){jt===Ot.pointerId&&(jt=-1,Gt(dt.MOUSE_EVENT_RELEASED,Ot));}return document.addEventListener("pointerup",p1),d.addEventListener("click",Ot=>{Gt(dt.MOUSE_EVENT_CLICKED,Ot,Ot.detail);}),d.addEventListener("pointerenter",Ot=>{Gt(dt.MOUSE_EVENT_ENTERED,Ot);}),d.addEventListener("pointerleave",Ot=>{Gt(dt.MOUSE_EVENT_EXITED,Ot);}),d.addEventListener("pointermove",Ot=>{jt!==-1?jt===Ot.pointerId&&Gt(dt.MOUSE_EVENT_DRAGGED,Ot):Gt(dt.MOUSE_EVENT_MOVED,Ot);}),()=>{document.removeEventListener("pointerup",p1);};}function bu(d,ft,dt,jt,Gt){let p1=typeof navigator<"u"&&navigator.platform==="MacIntel",Ot=t1=>p1&&t1.metaKey||!p1&&t1.ctrlKey;function at(t1,g1){let T1=D2(g1,jt);T1!==null&&dt.fireKeyEvent(t1,T1,g1.altKey,g1.ctrlKey,g1.shiftKey,Ot(g1));}return ft.addEventListener("keydown",t1=>{Ot(t1)&&t1.key==="c"||Ot(t1)&&t1.key==="v"||at(jt.KEY_EVENT_PRESSED,t1);}),ft.addEventListener("keyup",t1=>{at(jt.KEY_EVENT_RELEASED,t1);}),d.addEventListener("paste",t1=>{let g1=t1.clipboardData.getData("text"),T1=Gt.fromText(g1);T1&&T1.getAllAtoms()>0&&dt.addPastedOrDropped(T1);}),()=>{};}function D2(d,ft){switch(d.key){case "Control":return ft.KEY_CTRL;case "Alt":return ft.KEY_ALT;case "Shift":return ft.KEY_SHIFT;case "Delete":case "Backspace":return ft.KEY_DELETE;case "F1":return ft.KEY_HELP;case "Escape":return ft.KEY_ESCAPE;case "Enter":return ft.KEY_ENTER;default:return d.key.length===1?d.key.codePointAt(0):null;}}var zr=class{constructor(ft){this.canvasElement=ft,this.drawContext=new $n(this.canvasElement.getContext("2d"));}setDimensions(ft,dt){this.canvasElement.width=ft,this.canvasElement.style.width=`${ft/_A}px`,this.canvasElement.height=dt,this.canvasElement.style.height=`${dt/_A}px`;}getDrawContext(){return this.drawContext;}getBackgroundRGB(){return 16777215;}getForegroundRGB(){return 0;}};var wu={"chain.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB2UlEQVR4XmNgGJmAkYGJgQULZGXAhCygIGICQg4GdjTIARRDh5xAEQwNKYzXeK4xY1Oez3SNfz8TmoYUxq+6m9bfqLjDI4tmfhzLff9VR17FoWhwZXpkOm2v1C+79+cbD4gga8lneuhrfdv6/+ItSBpSGB+bTDos9IfzP9N/k3cX6+7wwByWz/TWR/E1w/+kO+8j4BpSGL9rz9wh+Efs14TF3heY/tu/vlF2hxtkSxzLY2/FVyz/om69jT0G8kM6QyNDM9MT41nbZX9y/J/ct1/4iU7aIYb/9q8uVx0UbmF64mV7i/1/0s0HSVM5dRgYrjB84/8U+bh80iH+34z/17Xc51ZhUGX8qpkK1GL6+krltyClVyz/c+69C57CwQsK1v+Sl9rdvgr8Z/7P9WvKxNs8fsD44ABqeWLkdYHhv9sL8e+s/xJuv4k4yajEAI6H/+6llxn+g6DUh75gSXhgugEdmXIUJB5150HSFA59oHKIBteGSyBhpv+q9wsVQaZDICdDKuNnneRjCY9exU3l5AUrh2gQelAQ/8Tgve2Tb25NjIgYBkmnMl5Sumi+hQmiGKqhgaGB57Pts4Bfek0sjkjmIxQhs8BpiZFBh0EPCLGlTnQxcGod8gAAp9es6fCW8G8AAAAASUVORK5CYII=)","eraser.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAC1UlEQVR4XnWU309SYRjHn8OYTTM3zTmjC8NFFBR10dJu27wx11VX/QNdeGebTsvua93mRbUmLhURzYgKRZRJUJLibLPpRDCZC3/b1AMc4Tw97+EAR2e8FzD4fr7v53nPOQDCDkT/s2KAKiwROOXvFP8JLnDDGHhgArzgg28wCQGYgiDMw6E+1pY2rsECLEIIIrAC1O2CV/AWzPAOLGCFQbDDJ/hC+CQktU5bXWJoMGlaVv0iPAy/GeCmeBf0QB/YYECKO6kkzAl6Z/+t+BmsSJotgmmN4lFYZcA4tfdQez+1D8Fnio+S1oHe5TAcFuJHVBNit4nGLYrHGOCBbin+ntodMEzxKMkMWy8LpfgSEUN4niC7TTBtqzYYMCG3Z9xHadgD/bC1hi+U4uw1TkhFsqs3adpngJdGHSAZB8VHYAZ4vdt+RSjED3KcvYWYWMJuFY0E+MjdQe5MZgYE7YhFL8soCJzDU1iKwdcE+HOjBoG/NNJbq5DJI40EPNiIPCLgO/Wzk4lS3D2kPyaTQe5hETZsLDfxRRQLkLuLDjJd7u8wJs/mRs22J7CR4re3Qq3xIpENPUUHG+biuqX2UVsVX41epTp9ZjINm0ttqTIEZECQboK0ztd/Ya951t+nRi2GFYgksxluSlO7BMTYLVblMVcmOOTEZ9N+i4YvkREmAySz2JaNE4CqtM5jvrZHcbbE5lmLU8NrJTEmc3drsT1VlmmXdsCbgTdSu7wKUi2zPqsaq7GWZCjeJBZnwxlAN9dRyecBtsvzH/4+DV+MNdsLT4/GJSU0jHde3T2KtMxYnA8X5tvT5cp2eehDLnVxrPPoLgWpx4Hd+8Lp/Zx7boYoe/g40eDtViIF6Se+xPX8qPl96DqECImrRMOY2fg3e1atwd16UX1cR75wEXr4VmGdxL52atbpX2HnxfTenfiJcelKr9CjvQp/IMWJ57AO62ndOKk7890/iSWBU4XaZc0AAAAASUVORK5CYII=)","fist.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAABj0lEQVR4Xu2S3yuDYRTHz7IZsfdCUrYLG4VCpJV6d0mS/A1zg79ArRArP4q5s4ulVousFCn/gX/g5WK7wkZMzIXMxjZ793XeMdn7vpM/wDn1PD1P53PO95znIfq3rwmYyU5tZPr7PJYoTjGaZ8xB1t+xZQqyP60iAEeWsThFGKpqbiE9CRECrnCHLnTAj17QbLV4E3mFXALbcOAGSXRiClnMgZLkZR/QYg0UEpDBDuxI4IHzT6OADdRiEK2gE12gBWmWE8Yj7weQIGOT650zaoiqAVtjDx2KeEWR8xZ4lUt+jEWkWJxJBYgkWa7NLy5WXWl5FilzhfoKwEAzzkwYI3DhTQUoVYpYR10FYCTPaDbJ2heQUwGfR0kFELmt6Utu9Jmz6dmFpmmR7v141w1XBiCBNFNacyKmCxT59kgHGBIQ4g7UkpRzESuwaCoQ+YZxqwGUGaUwrvvStqazLeRLj/bTZUTQB5rQfg1jjac9e8qAzFgZKbDMALpj1Kz3Z0WKjiGIfex++x6H98PgK4d/APiDpdwStyBoAAAAAElFTkSuQmCC)","hand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAAB50lEQVR4Xq2TwUsUcRzF3yy52TboHiKVLBCElb1479g5kgKPCkKs4ME6eAiiP0GELileStew6NLJq5Sihw6BG2VlECymB2PVXd10Zl7vN+OuOjNLl35fZhiY+fzee9/vb4D/tAow9fafu90BsqrvnXzNBwSTTBAZ2I3BkSbeYBMzLHNRn9/nEFHCvYZAutDHDfb5wDvanOVnoozBhkBzIUeHOQEHfO8DW0QV0xjC3VjoUh2onAC/ZPAme4giemMQo+BKoaeusCmdSa41MmYL8HxLgUKe27q/UKLzSSxcwUOMwk7VgTKX6sBz7oeA61hJHSUP8ST1raZQA7Z8hb0QkO1WE+d5lZdlx+UbTsjSR95i0bc0E7GUzfIHdziouRqFPzwU5ii4IzOP+SFiqe3i8jyP1A1bgKPyVGa5/M2S7iaDdWZ8KUwNe1Uus923FHx8ujwZPA9YGO48+CK/Ywp4un8N8SIZgG78fCTvFV1n9w+ePe6GFMzMJ7u4zuOInQDYYaKcDB3Ba63FGQWPLk8mt2MAWKv96kk0sKM2TBFfkQ4fvkx6fyHSH9NalwO0PkXPaktirt8rhRpqelblbV6IASzkmivP9NrEdE9AM/FXbCPG4v+6p+3VcZ2ql5zTZSqv+XcQ6+gKgL8a/RiIJPdySgAAAABJRU5ErkJggg==)","handPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACAUlEQVR4Xq2Tz0sUcRjGn1lytW2wPUQamSAIK3vp7kk6i1LgMUGIFTqUBw9BiPsPhNAlpUtuKhpdOnVUKqlDh6CNsh+CsJQdjC13ddOZeXq+M7uTzu7SpXmZLwvzfr7P8/5Y4D89eZh48s/bhoC04nMXH/EmwThjRAp2c/B6C7vZwhRLXFP6NY4RRVxpCiTzw/zCYR94RpsP+Z4oYbQp0JbP0GFGwB6f+8A2UcF9jOFyQ+hkCJSrwDcZ7GcfUcDFBohRcKXQFyp8lc4s3zYzZgvwfEuBwgK/65xXRccrsXAGE7gBOxECJb4IgQfcjQAX8DJxEN/HVOJTTaEGbPsKvyJAuldNXOZZnpIdl485I0tveIkF31KuzlI6zU3ucFRzNQq/uS/MUeGOzNzm6zpLHa3ryzxQN2wBjsIT4JE6f7Co09RgHRlfAnPjXoXr7PQtmdSsNqn2eDJ4HLAw3rX3QX4nVaC534sA0bYCvdi6Je9lveb+mkLw2+PPiIKZ+WwPN3ioj9MyczQMsMNYKR5ZwfOnCzkVTq7KTpYDgrKCp5XuyGw9AOvViHoSmPhryaS7nCM+IhldvlRy92nYmaBLQWtdXqX1rn5X22OLI16xihhjBjA9q3CQJxoAFjJt5Xv6bBLdKmgmvsIOYrLxv+5uZ+WOtmqJi3pNLGj+54gN9ATAH0reBFwT+FQbAAAAAElFTkSuQmCC)","invisible.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACjUlEQVR4XnWTS09TURSF178wMUExvhIfiY+0Q6MThwwEHYk6MODAqHHgzD9gHDgWYhQBAUWMqY/SogSkCGJJNAiFUiyV2qqURkFa7cPvnJZiBXMH7Tl3rbXX2ntf5bWgyH+emPI8v8reAn8nj3r0Qr3qU78GNKghDWtEfk3ot2LKKq6AphTUjGYF26ObuqU7alabOvRQj/VEz6APKS23jvI/rY/6AD2ksCH0AG9Sq9r1QJ0W/hyREFbcOqQNcqiL/3FuIpozhJeot6J+H/VHegrci60lSA7t0VVtlRNqTvPAY4bQqxYL70LdxSsvd2l164B2qVHLVN4L0U2VhL4aQl9RveDdS9gl4IcBNWpRUShNUJxUT3OG0E/UTo4u4N0a1U9SHdRu3QOQIOw8N+0Yc/A2ZwgDeHfh3ZgZpbBH+zDTAPy7Jok6qU/K0Jht2k91CL5SVD9aHh1BvYHiCxYe1nV9w9h5bVcdk4DwGn3TmYg1Y9Rb8P2FQYVQbOM+qjPaDDwMguMw3j00Mst0narUDaUgBICPI5VgAhe5rUVguZBhhMaGOEzrrU6pQpfpdoqeB8k2hvoF7UB9mhxmsyD4WYIs0c/ifEKnoVxSEsptQke5rdQ5vGdRt4SYXTEf2lWq1l1s1GojlDiAgpmTLN4KHEKegw+NKvsco8oglApd0XtoOzEzhZmCuq2QJ7RRX3lqmO84xjaxGltUj/dVcJEwhuIqwVRptsZMZwL/wG2FPIHryijVVBnUNdJlS1ZKlmJEzkApr1JDj5KsyeJaQsR+fDlW8G/KCWyl1tgpziEIxczwVcnYcT6p5LpwO7gZ5jzH7mT0hmbW0+IO/UBixXX5L4RZlmpOn8u6vT7Y3P4BKLGmkBK1qvgAAAAASUVORK5CYII=)","lasso.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADg0lEQVR4Xk2UXWxTZRjHn7XYfWhh4DKlIFLRdlucImbGr2gEwYwsAjGRXXihRlTOFrOoF0RcdWwhEVnQdgOmbgzoliEhAZJlvZwdMQg6No/U2IYOWVrK1q1d19q1Pef8fU7bsy3/nIvznv/ved/n47xEZKDivNpK8RoE6XC4J9gX7A/2hXukwxCwta1UcxhUu4PcLFHnqZnaf+/E0FhTqDpiiaqqjjSFhsbvnZja76kRdarLQRzbTSjAuozgczaHVoJYhShX1krlSmH2bSWaQz5nRsA6MJQDrHMtx0Y3y4QS7Eo0eTpG3K5rP7tdHSNNnl2JNQxtlo+NzrXAmgXEDcnWA/5ihbAtOeyKtKf3wmIvs6+2l8GS3hv51u3aliQUKwf8yVZxA2GNIvSIBCNaJmPt/jrR6KZfyL6omQdQF2tvmTTyPt2iIhDq/f1mth/1LtjCVTa9QAIN8dEl8tK8mgJBj6oF21GvEWb4+wlft08Sds+FHbPmDl2A/mOl2Ran7SQylkV0MIcdu+cI7J3utUb0aL2q1DZSlDiRvObpKdpCZymVf1dqW6/qYY1QsM8SXYULw3jcq8XLWlSghOrJpYUwXxpeDUuUZn6smNWheUx6l7uxGD8HrKXz9A2doQidJnnPoWt6VEQItm4/4aX0LSd2wLCEqMB6+otmqYt8K1LVd3+oTRA6/yXUTXe+mCHsjE048QbWw8R6EHoNAMlGbA+d/PSOAZXKZC9tKbr19m+DFZztc/E/B2/3jp//x5logyC/8oyBd9DBIjeODj4/X8hTcO4PZZ/a6RKpYfBKWb6CuTQ+i41elD5A5cLHga7jPjOvWDIDv6e/EE25WXo49N0mPmF21OR3wm/FTahMf3/dNtLoe4GPa1LeC9y4xGUxacNnvHnykfTG+IWxJzOlyhnPzHH7bUJRvgINiSuu+CE8i2Jw8NwORq/jidRDktMz4COU46twnVyEx+brwxfFQDc+weu4P0drQFHq/Z+uP73UZq7vh8FAV6oeL2OTZl4OFMCUaZw4NXDjyNTBWENylaLDwXFpH+5Tm5mku4sN1XZQFwrkR+OvyjvxpvT5kTuEHcmpDh5i/qIQ/1l5LQd4RifUOS1A1XTnRq76r5dhzRlTPBxx7vlSDstmiA0r8NGXUcLZm6jJARkGZvnJAtqtod4JmhJbT/9tiZ4bS9QsX83dGv8DIrFVpnd+f4UAAAAASUVORK5CYII=)","lassoPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAADiklEQVR4XlVUbWxTVRh+12K3oYWBy9SCSAHbbXGRaGb8SoygmJFFICayH/5AIyrdYhb1BxFXHWswQRa03WVM3RzYjaEhYcZl/Tk6QhB0bN62xjV0yNJStm7tutbetvfex3N7e3XmyUnuOXme9+O8zz1EZKDyIhwVeBE28VisLzIQGYwMxPrEY7Bhh6NCYxgUuou8DLwuUD936G736GRrtC5uSSioi7dGR6fuds8dCtTzOoXlIhbbSyjBhrwt6G6LrgExlKJKfkiskksLuzVoiwbdeRs2gIlUgXWp/eTEdomwGnvSrYGuca/n2g9eT9d4a2BPej0TbZdOTiy1w1oQ8JsyHYdD5TJhZ2bME+/M7YfFWelc56yEJbc//oXXszNDKJcPhzId/CbCetnWxxOMaJ9NdoYaeaOXLpHg8HOCw0lOWrgPjcnO9lkjy9PLyzZCU2jQzOgnpgV7rNaut5GNRsnPEXzcstICQY9awX5i2ggzQoOEzzpnCXuXYq5Fc5cuTH8z5IoCnkRVooM55tq7RGDc+X5rXI+Oq3JDCyVILhTj54bPEYbPKV+CoyAhuaHjqh7WOEUGLIm1uDCGbdOFeEoxK+Hjilvz8Ng6WBK08E31og5tk+IBNg3GFBw+zlfMoHwJjrMUpzMk7Tt6TY/qOMHeGyI8l7vpxi4YtOBq0xvJR4vUQ8FV2bo7XzekCdxfhMZ57tk8YXdyxo1XsBEmhvv93ZoAJBnxcvT0B7cNqJFn++mJspuv/zJSzcb2VOr3kVv9Uz/+6U47BI/vZ+FzlkEHi9QyMfL0cilzwfnf5IPKpFeLzSOXK4s3qDb8YXLiovg2aoT3wj2ngmZ2YskP/Zr7mDepXnow+uVWVmHBatIbsddSJtTkvrpuH28JPsPKNclvhm8Ms2sxaeYz+k8/nNucujD5WL5CPhtYOOW8RSgr3kBz+rIndRRPohwsuJrBOO16NPuA6A4MBQlV+DTWKJVhy3JT7CIf7sX7eAn3qmpNUJZ969vrj7PGNRjwTiTck23C89iqkVcKSmDKt8x8N3Tj+NyRZHNmrazDkSnxIO5RhpmhO/9G0jIoByXSI6kXpN14Vfzo+G3CrsxcFzOx4iJif1YRKwXMozNssZi189xmdutXfoJVJWaZOVJs5v/18H+/rcK7nyQI3/tRrwryTLDIVkGgvRrKm6AhvePMH5bE+cl0/cpT9dX4B08ibwJFFp9KAAAAAElFTkSuQmCC)","pointingHand.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAADe0lEQVR4Xq2V+yvrYRzHnWwuaaUst9mY21iyHcVySghxJJLLDy7L2Y9S4geHThx+5ISc0vCDf0AS6fj1rPxkRJ2dJjvtjNolt1y+NMY+5/N5yonZbIunvm179nyf1/N+fy5PSMgLo6OjwxAWFgaxsbHQ2NhonJiYELy0Pqj/CgoKPqSmpjoGBgZgfHwciouLna2tre+D2uSlxXK5XJ2UlMTt7e2B2WyGjIwMiImJ+fhmADy9OjMzk7PZbHB0dAT4HXp6en6+GSA5OVktk8k4u90Ox8fHTEF3d/fvNwNIpVKm4AFACgYHB//Mzc0pUZ3o1SBPgEKhgKysLFd5ebkdlekTExMLUKUU56VVVVVhQQM9AdnZ2dDe3g5bW1tQU1MDKSkpNrTtr0qlsvT3939/NSA9PR2mp6fh7u4OhoaGWG2gXdDZ2Qnx8fFWBH3CR4P1IvMFe4feqoqKir7k5OQoSQEF2Wq1siwiwMzMDLjdbhgeHoa8vDw4OztjiiIiIqCyshLq6uoA39mRSCSKZ5DQ0FAxnuQXeutG+Yb8/Hwt1gIDHB4eMsDs7Czc399TsAFPCxzHwe7uLojFYlhdXQWDwQBKpZLDfdTeVMhra2vBaDRCV1cX2zAtLQ0IcHp6yuYWFxeZRfPz89DX18cAVIgYbNje3oaTkxPAJODwsN4BLS0tTPbGxgagTBCJRECFdnt7C+fn53B9fc0UXF1dweXlJbhcLnA4HAxmsVhYvVRUVHA8Hu85IDIyUl5SUgImk4mduKmp6T+ANiXvHz+Ag+ZJEYHpkxTgHt4VCIVCCTY009TUFNzc3MDk5CTlPRwcHLCN/A2CE6C0tNQ7IDw8nI8Z8LWhoeGWFGxubkJvby+THSyAz+d7jUEIpp46ISGB0+l0zHeKh9PpDAhACimd0QUuKirKO4BSC3uOTqPRwMXFBQvig/+BWER9C+8RLjo62jcAPayhvqPX6xmAgkf++hu0huKF2cdhivsGjIyMCAsLC81jY2PMHgIEMgiwvr4OcXFxHNaCbwDZ1NbW9q2srAz29/fZ6QOB0LrR0VHAdPd/ZwgEAhF6uaPVap/EwbMOHpTRPGUbKgfsBv5vPawJXn19/WfsR04qPMoozzg8hlEirK2tMUBzc3Pg93Zubq6uurqa9aCVlZUnz/LyMvu9tLQECwsLgPVD1f/Ds8n9AyubuM4FiwqZAAAAAElFTkSuQmCC)","rect.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACc0lEQVR4Xo2UT0hUURTGf7uipMy8GbUwigx0WbiwpCgQKmgl1CYj3Plahtg/eguFUIoamNxUCEFCGlIgEhSORmQRNeVMSkhpWcSU1KSOQ9i8vvfmpiMZOB8Xzjv3fPPd8+dertPHQICX/OAVnTRTz1GOaNXL7pQvSdTG9KHwCVJMyzlMFw1UsJFC8lmtVSi7Qr4u7SUVk1KsmCkyZtJETchUmTKh3NSYk6ZBq0a276nSXtRMGc+kfMIkiRY3shQkWiZ9wjjxMN5SEAuPQ5qegBALe07a+eiMCj9lxZ0ZJ+N483A9oQf6qbWED6rITW4IcQGaGJuT/Y0f7rm1cJliS+jnLjNkAviEDXTwVbZ/2gwjobLESKgY6iiwhI5gyw/N/T2zKtljF0CN6p39SAZbCVyhTpwVLKOUM8wGmn4lEy35yGH+Jp1TKF9nrRp3mzbOMcp9m0MhtLPrvwRDr/6/mwda6cZ4ON24Hd5z1RISfGNIAdmDnWabtHtlzxLjtY1pgucM2o8rPOUhj+UZY0qDWGEJad5y0ca8mO90LLyKA+yjkirN6QWeaO0kwjvucZw8m0MwGinLzmelUi1nP3vYQrUCH3GNY5Qp+UpbpelcQjNnWcchVWRQhc2jiPWsYTl7aZRetvTBtP5VaJW7la0c5g4npFbCbk5xSzlNKPGRUKk6vYCwScIH2ax2lagH1VJ6wxdbNVUumKUFhGFNaMxiyEk4v3Jn1VmE4LluBK804W/5zdDA/+OxCtk58dzeNjcS7vYJ/v1bzGMJnm7rPDLmkzlvdph2E5/zRs13M2BS/p3OvhoLkVSBi7jE5zn/VPCy+K/GH86K+gV5WsL6AAAAAElFTkSuQmCC)","rectPlus.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACi0lEQVR4Xo2UT0hUQRzHP6c2KmrTnYoKjCAP+45FB0uKAqECT0FdMsKbr2OIUdE76EUpamHzUmoEGWmIgUnQn9UOWURp7ksJKSuLeCW1qesStq/fvB11lzr4vgzM/Ob3md+b3/xmuEYfA4Fe8ZMhOmmklmMclVYr/U6xpRg0Pn2I+yRpZsQ4Shd1lLGJCGHWSItIv0xsXTKXEp+0+AqZJqum1KCKqQpliXaqKnVK1Umrkr62VMjcoJpWvkprYAqvyUksRV7TlAYmcOP4S1EyPgEZegMgGfftjP3JHhf9kp5rz9pZ21+U44t6oZ9qA3yUjNygReSKoIEPC2H/oN19pxouUWKAfrqZJRtIAxvp4Jv09d9mGYtZ3lisBGooMkBHMKVd5QsRlSbfcxMl99tFUCX5zg1SwZSHI6qJ4q+wNHWGuSCmzqTXFEYMan7TeYlyBSi2ItymjXOMc9/sIQLt7C4AMiE36ka7K/HDlUXR1uh0qIcH9JCpd+OZ+u3wnisG8PjOCEOydr6S1hxJXhufBnjBsBlc5hkPeRS6bt2zbuUiWK1WJpThLReMz8vFk07GV3OQ/ZRTIXWqN11t7SLBO+5yglVmD0FppA0dZqWkcScH2MtmAVqsJ1zlOJZUbbnJ0kw+0MhZ1lEpGRnGCS2zVGgDa1nOPup5ahYNqnU+QrOYm9nGEe5wUqKVsofT3JRdTspZjMWictIFwBYJfIitcsClFHNYIr3hq7iapAW1VACMSoUmjUZsz/6dX6t2rvgKAN9xEvhRT0/oZaXg/7EYIFcnvvO4zUnEezSg79//LAbw5bYuKqs+q/Nqh2pX7oJ1UP1QAyqt73Tu1ShUShK8not8WbBPBy+LfjX+AjC0Avv8MQHTAAAAAElFTkSuQmCC)","zoom.png":"url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAQAAABKfvVzAAACiUlEQVR4XpVTTUgUYRh+nJWCxUsNevBiYGm7UjAnFyoI9ORKSxt4a71Ei9I10EPg0VO7hOE1ikQSofXiYv6QRX9bkTAjbbM/B2mz8id3Z2f/ZvbtndR0NYn9XuYbPuZ55n3f730efMQCZjCLOX4/xyJe4BXe4C3e4QNi0JDCBu8JRPkURxIMm8BjjOMJJvEUU8jadUn36D79Zs5X8JhSyb6AX/iBzxxfLMIMwx/gIcYwXRMTNa86Mhrp23QXOwx3sW9zNKKOaF5NzAlrnCWBFXAx43jE8PeC0b4U9KstJmh/tJh+dSlotJeEDYZ/Bea5GAtudoZDkm5jcCP1poejd+XhaG+6kc82kvRwiDpLwha+A1RX8K4OaDfCU038UTQGk/GA2qO4lDbFpfbEA4NJ0QA1UThktOeFDBPOLd+/qF9PtOZBDi30LNU91FBfgz+rvmaoIdUVmnZo4CxLQU3cYsLJ7G1HebvqC7nIhHJ6G7y7mmsNKTQtGjbyq5p3Dnzv2au31i24QK35cEhpriQAzbUp92AS1GKqI7odL5F1jM9bzV7KzoyVLvfbDhKAoYZ4wGp/NFKU8Br6scS9VrM7Fw8a56/8A271ovb0pkF9m2UPy2AWGd/k4sYdpal/7+fH4cTxvaPsGo6C3EXysWYWQafIR2cqSnFyV237CM6ADOowyM8Si+xkkCsztB2RIYZCFT2QB1mYVdwSSSza/N85nD1qDl27cyA76EQ1k6bqtESiRagrXFsdyBxQq+ySnfIhtbJ6gHX2bBprQvmAHwKH/GDBmRCDyibP4Od/HLcNZ0KcCSnkmLQsKEd4mmvfgTMhybGCb0zI4RPb1bSbUtnDUvHz4yGJ7BUWx2+5Iu7Rc33AhQAAAABJRU5ErkJggg==)"};var H8=Object.create(null),gu=3/4,Jr=class{constructor(ft){this.HOTSPOT_32=ft.HOTSPOT_32,this.IMAGE_NAME_32=ft.IMAGE_NAME_32,this.cPointerCursor=ft.cPointerCursor,this.cTextCursor=ft.cTextCursor;}getCursor(ft){if(H8[ft])return H8[ft];if(this.IMAGE_NAME_32[ft])return this.buildCursor(ft);switch(ft){case this.cPointerCursor:return "default";case this.cTextCursor:return "text";default:throw new Error(`Unknown cursor: ${ft}`);}}buildCursor(ft){let dt=this.IMAGE_NAME_32[ft],Gt=`${wu[dt]} ${this.HOTSPOT_32[ft*2]*gu} ${this.HOTSPOT_32[ft*2+1]*gu}, default`;return H8[ft]=Gt,Gt;}};var Kr=class{constructor(ft,dt){this.title=ft,this.rootElement=dt,this.elements=[],this.dialogElement=null;}setLayout(ft,dt){this.hLayout=Bu(ft),this.vLayout=`${Bu(dt)} auto`;}add(ft,dt,jt,Gt,p1){this.elements.push({component:ft,x:dt,y:jt,x2:Gt,y2:p1});}createTextField(ft,dt){return new q8(ft,dt);}createLabel(ft){return new E8(ft);}createComboBox(){return new V8();}createCheckBox(ft){return new I8(ft);}setEventConsumer(ft){this.consumer=ft;}showMessage(ft){window.alert(ft);}showDialog(ft){let dt=document.createElement("dialog"),jt=this.rootElement.getBoundingClientRect();Object.assign(dt.style,{position:"absolute",marginBlock:0,left:`${jt.left}px`,right:`${document.body.parentElement.clientWidth-jt.right}px`,top:`${this.rootElement.offsetTop+30}px`}),this.dialogElement=dt,this.rootElement.getRootNode().append(dt);let Gt=document.createElement("form");Gt.style.display="grid",Gt.style.gridTemplateColumns=this.hLayout,Gt.style.gridTemplateRows=this.vLayout,Gt.addEventListener("submit",t1=>{t1.preventDefault(),this.consumer.fireOk(),ft();}),dt.append(Gt);for(let{component:t1,x:g1,y:T1,x2:d1,y2:o1}of this.elements){let K1=document.createElement("div");d1===void 0?(K1.style.gridColumn=`${g1+1} / ${g1+2}`,K1.style.gridRow=`${T1+1} / ${T1+2}`):(K1.style.gridColumn=`${g1+1} / ${d1+2}`,K1.style.gridRow=`${T1+1} / ${o1+2}`),K1.append(t1.getElement()),Gt.append(K1);}let p1=document.createElement("div");Object.assign(p1.style,{display:"flex",flexDirection:"row-reverse",gap:"15px",gridColumn:"1 / -1",gridRow:"-1"}),Gt.append(p1);let Ot=document.createElement("button");Ot.type="submit",Ot.textContent="OK",p1.append(Ot);let at=document.createElement("button");at.type="button",at.textContent="Cancel",at.addEventListener("click",()=>{this.consumer.fireCancel(),ft();}),p1.append(at),dt.showModal(),dt.addEventListener("cancel",()=>{this.consumer.fireCancel(),ft();});}disposeDialog(){this.dialogElement!==null&&(this.dialogElement.remove(),this.dialogElement=null);}},an=class{setEventHandler(ft){this.eventHandler=ft;}fireEvent(ft,dt){this.eventHandler(ft,dt);}},E8=class extends an{constructor(ft){super(),this.element=document.createElement("label"),this.setText(ft);}setText(ft){this.element.textContent=ft;}getElement(){return this.element;}},q8=class extends an{constructor(){super(),this.element=document.createElement("input"),this.element.type="text";}setText(ft){this.element.value=ft;}getText(){return this.element.value;}getElement(){return this.element;}},V8=class extends an{constructor(){super(),this.element=document.createElement("select"),this.element.addEventListener("change",()=>{this.fireEvent(2,this.element.selectedIndex);});}setEnabled(ft){this.element.disabled=!ft;}addItem(ft){let dt=document.createElement("option");dt.textContent=ft,this.element.append(dt);}getSelectedIndex(){return this.element.selectedIndex;}setSelectedIndex(ft){this.element.selectedIndex=ft;}setSelectedItem(ft){let dt=this.element.options;for(let jt=0;jt<dt.length;jt++)dt[jt].textContent===ft&&(this.element.selectedIndex=jt);}getSelectedItem(){return this.element.options[this.element.selectedIndex].textContent;}removeAllItems(){this.element.innerHTML="";}setEditable(){}getElement(){return this.element;}},I8=class extends an{constructor(ft){super();let dt=document.createElement("label"),jt=document.createElement("input");jt.type="checkbox",jt.addEventListener("change",()=>{this.fireEvent(3,jt.checked?1:0);}),dt.append(jt),dt.append(ft),this.element=dt,this.checkBox=jt;}setEnabled(ft){this.checkBox.disabled=!ft;}isSelected(){return this.checkBox.checked;}setSelected(ft){this.checkBox.checked=ft;}getElement(){return this.element;}};function Bu(d){return d.map(ft=>ft>0?`${ft}px`:"auto").join(" ");}var qe=class{constructor(ft){this.imageData=ft,this.dataView=new DataView(ft.data.buffer);}getWidth(){return this.imageData.width;}getHeight(){return this.imageData.height;}getRGB(ft,dt){let jt=this.dataView.getInt32((dt*this.imageData.width+ft)*4,false);return (jt&255)<<24|jt>>>8;}setRGB(ft,dt,jt){let Gt=jt>>>24&255,p1=jt<<8|Gt;this.dataView.setInt32((dt*this.imageData.width+ft)*4,p1,false);}toDataURL(){let ft=document.createElement("canvas"),dt=ft.getContext("2d");return ft.width=this.imageData.width,ft.height=this.imageData.height,dt.putImageData(this.imageData,0,0),ft.toDataURL("image/png");}};var Zr=class{constructor(ft,dt,jt){this.canvasElement=ft,this.dialogRoot=dt,this.JavaEditorArea=jt;}register(ft){this.javaUiHelper=ft,this.cursorManager=new Jr(this.JavaEditorArea,ft);}grabFocus(){this.canvasElement.focus({preventScroll:true});}setCursor(ft){this.canvasElement.style.cursor=this.cursorManager.getCursor(ft);}showHelpDialog(){}createImage(ft,dt){let jt=new ImageData(ft,dt);return new qe(jt);}createImageFromBase64(ft,dt,jt){jt=jt.replaceAll(/%\d+%/g,at=>"A".repeat(Number(at.slice(1,-1))));let Gt=hu(jt),p1=new Uint8ClampedArray(Gt),Ot=new ImageData(p1,ft,dt);return new qe(Ot);}createDialog(ft){return new Kr(ft,this.dialogRoot);}runLater(ft){typeof requestAnimationFrame=="function"?requestAnimationFrame(ft):typeof setImmediate=="function"?setImmediate(ft):setTimeout(ft,0);}};function ku(d,ft,dt,jt,Gt,p1,Ot,at){let{readOnly:t1=false,initialMode:g1="molecule",initialFragment:T1=false}=ft,d1=document.createElement("div");d1.dataset.openchemlibCanvasEditor="true",Object.assign(d1.style,{width:"100%",height:"100%",display:"flex",flexDirection:"row",alignItems:"start",backgroundColor:"white",touchAction:"none",userSelect:"none",webkitUserSelect:"none"});let o1=d1.attachShadow({mode:"open"});o1.adoptedStyleSheets=[cu()];let K1=null;t1||(K1=document.createElement("canvas"),o1.append(K1));let Q0=document.createElement("div");Object.assign(Q0.style,{width:"100%",height:"100%"}),o1.append(Q0);let $1=document.createElement("canvas");$1.tabIndex=0,Object.assign($1.style,{outline:"none"}),Q0.append($1),d.append(d1);let H0=new p1(new Zr($1,Q0,jt)),qt=new jt(u2(g1,jt),new Gr($1,dt),H0);if(T1)if(g1==="molecule"){let e1=new Ot(0,0);e1.setFragment(true),qt.setMolecule(e1);}else {let e1=at.create();e1.setFragment(true),qt.setReaction(e1);}H0.setEditorArea(qt);let HA=t1?null:new Gt(qt,new zr(K1),H0);function PA(e1){$1.style.width=`${e1.width}px`,$1.width=Math.floor(e1.width*_A),$1.style.height=`${e1.height}px`,$1.height=Math.floor(e1.height*_A),e1.width>0&&e1.height>0&&qt.repaint();}let Ve=Q0.getBoundingClientRect();PA(Ve);let R1=new ResizeObserver(([e1])=>{PA(e1.contentRect);});R1.observe(Q0);let m0=null,pn=null,yt=null;t1||(m0=S8($1,qt,jt),pn=bu(Q0,$1,qt,jt,Ot),yt=S8(K1,HA,jt));function H1(){d1.remove(),R1.disconnect(),m0?.(),pn?.(),yt?.();}return {editorArea:qt,toolbar:HA,uiHelper:H0,destroy:H1};}function u2(d,ft){switch(d){case "molecule":return 0;case "reaction":return ft.MODE_REACTION|ft.MODE_MULTIPLE_FRAGMENTS;default:throw new Error(`Invalid initial mode: ${d}`);}}function du(d,ft,dt,jt,Gt){class p1{#t;#A;#r;#e;#f;#P;constructor(at,t1={}){let{editorArea:g1,toolbar:T1,uiHelper:d1,destroy:o1}=ku(at,t1,K1=>this.#i(K1),d,ft,dt,jt,Gt);this.#t=g1,this.#A=T1,this.#r=d1,this.#e=null,this.#f={[d.EDITOR_EVENT_MOLECULE_CHANGED]:"molecule",[d.EDITOR_EVENT_SELECTION_CHANGED]:"selection",[d.EDITOR_EVENT_HIGHLIGHT_ATOM_CHANGED]:"highlight-atom",[d.EDITOR_EVENT_HIGHLIGHT_BOND_CHANGED]:"highlight-bond"},this.#P=o1;}getMode(){return this.#n(),this.#t.getMode()&d.MODE_REACTION!==0?"reaction":"molecule";}setMolecule(at){this.#n(),this.#t.setMolecule(at);}getMolecule(){return this.#n(),this.#t.getMolecule();}setReaction(at){this.#n(),this.#t.setReaction(at);}getReaction(){return this.#n(),this.#t.getReaction();}setOnChangeListener(at){this.#n(),this.#e=at;}removeOnChangeListener(){this.#n(),this.#e=null;}clearAll(){this.#n(),this.#t.clearAll();}destroy(){this.#n(),this.#P(),this.#t=null,this.#A=null,this.#r=null,this.#e=null,this.#P=null;}get isDestroyed(){return !this.#t;}moleculeChanged(){this.#n(),this.#t.moleculeChanged();}#n(){if(this.isDestroyed)throw new Error("CanvasEditor has been destroyed");}#i(at){if(!this.#e)return;let{what:t1,isUserEvent:g1}=at;this.#e({type:this.#f[t1],isUserEvent:g1});}}return p1;}function mu(d,ft,dt,jt){class Gt extends HTMLElement{static MODE=Object.freeze(Object.create({MOLECULE:"molecule",REACTION:"reaction"}));static observedAttributes=Object.freeze(["idcode","fragment","mode","readonly"]);#t={idcode:"",fragment:false,mode:Gt.MODE_MOLECULE,readonly:false};get idcode(){return this.#t.idcode;}set idcode(Ot){this.#t.idcode=String(Ot),this.setAttribute("idcode",this.#t.idcode);}get fragment(){return this.#t.fragment;}set fragment(Ot){this.#t.fragment=!!Ot,this.#t.fragment?this.setAttribute("fragment",""):this.removeAttribute("fragment");}get mode(){return this.#t.mode;}set mode(Ot){this.#t.mode=String(Ot),this.setAttribute("mode",this.#t.mode);}get readonly(){return this.#t.readonly;}set readonly(Ot){this.#t.readonly=!!Ot,this.#t.readonly?this.setAttribute("readonly",""):this.removeAttribute("readonly");}setMolecule(Ot){this.fragment=Ot.isFragment(),this.idcode=`${Ot.getIDCode()} ${Ot.getIDCoordinates()}`,this.#A.setMolecule(Ot);}getMolecule(){return this.#A.getMolecule();}setReaction(Ot){this.fragment=Ot.isFragment(),this.idcode=jt.encode(Ot,{keepAbsoluteCoordinates:true,mode:jt.INCLUDE_MAPPING|jt.INCLUDE_COORDS|jt.RETAIN_REACTANT_AND_PRODUCT_ORDER})??"",this.#A.setReaction(Ot);}getReaction(){return this.#A.getReaction();}clearAll(){this.#A.clearAll(),this.idcode="";}moleculeChanged(){this.#A.moleculeChanged();}#A;#r(){this.#A||(this.#A=new d(this,{readOnly:this.readonly,initialMode:this.mode}),this.#A.setOnChangeListener(this.#o),requestIdleCallback(()=>this.#e()));}#e(){switch(this.mode){case Gt.MODE.MOLECULE:return this.#P();case Gt.MODE.REACTION:return this.#n();default:throw new Error(`Mode ${this.mode} is not supported`);}}#f(Ot){let at=Ot.indexOf(" ");if(at===-1)return ft.fromIDCode(Ot);let t1=Ot.slice(0,at),g1=Ot.slice(at+1);return ft.fromIDCode(t1,g1);}#P(){let Ot=this.#f(this.idcode);Ot.setFragment(this.fragment),this.#A.setMolecule(Ot);}#n(){let Ot=jt.decode(this.idcode,{ensureCoordinates:true})??dt.create();Ot.setFragment(this.fragment),this.#A.setReaction(Ot);}#i=false;#Q(Ot){this.#i=true;try{Ot();}finally{this.#i=false;}}#o=Ot=>{let at=this.idcode,t1=this.fragment;this.#Q(()=>{if(Ot.type==="molecule")switch(this.mode){case Gt.MODE.MOLECULE:{let T1=this.getMolecule();this.idcode=`${T1.getIDCode()} ${T1.getIDCoordinates()}`,this.fragment=T1.isFragment();break;}case Gt.MODE.REACTION:{let T1=this.getReaction();this.idcode=jt.encode(T1,{keepAbsoluteCoordinates:true,mode:jt.INCLUDE_MAPPING|jt.INCLUDE_COORDS|jt.RETAIN_REACTANT_AND_PRODUCT_ORDER}),this.fragment=T1.isFragment();break;}default:throw new Error(`Unsupported mode ${this.mode}`);}});let g1=new CustomEvent("change",{detail:Ot,bubbles:true});if(this.dispatchEvent(g1),Ot.mode==="molecule"){if(this.idcode!==at){let T1=new CustomEvent("idcode-changed",{detail:this.idcode,bubbles:true});this.dispatchEvent(T1);}if(this.fragment!==t1){let T1=new CustomEvent("fragment-changed",{detail:this.fragment,bubbles:true});this.dispatchEvent(T1);}}};#D(){this.#A&&(this.#A.destroy(),this.#A=void 0);}#u(){this.#D(),this.#r();}connectedCallback(){this.#t={idcode:this.getAttribute("idcode")||"",fragment:this.hasAttribute("fragment"),mode:this.getAttribute("mode")||Gt.MODE.MOLECULE,readonly:this.hasAttribute("readonly")},this.#r();}disconnectedCallback(){this.#D();}adoptedCallback(){this.connectedCallback();}attributeChangedCallback(Ot,at,t1){if(!this.#A||this.#i)return;(()=>{switch(Ot){case "idcode":return this.#t.idcode=String(t1),()=>this.#e();case "fragment":return this.#t.fragment=t1!==null,()=>this.#e();case "mode":return this.#t.mode=String(t1),()=>this.#u();case "readonly":return this.#t.readonly=t1!==null,()=>this.#u();default:throw new Error("unsupported attribute change");}})()();}}return Gt;}function O8(d){let{GenericEditorArea:ft,GenericEditorToolbar:dt,GenericUIHelper:jt,Molecule:Gt,Reaction:p1,ReactionEncoder:Ot}=d,at=du(ft,dt,jt,Gt,p1);function t1(){let g1=customElements.get("openchemlib-editor");if(g1)return g1;let T1=mu(at,Gt,p1,Ot);customElements.define("openchemlib-editor",T1);let d1=document.createElement("style");return d1.id="openchemlib-editor-default-style",d1.innerHTML=`
    /* dynamicaly added from openchemlib registerCustomElement with low priority */
    openchemlib-editor:defined {
      display: block;
      height: 400px;
      width: 600px;
    }
    `,document.head.prepend(d1),T1;}d.CanvasEditor=at,d.registerCustomElement=t1,delete d.GenericEditorArea,delete d.GenericEditorToolbar,delete d.GenericUIHelper;}function vu(d){d.prototype.getNextCustomAtomLabel=function(dt){let jt=dt||"1",Gt=new Set();for(let Ot=0;Ot<this.getAllAtoms();Ot++){let at=this.getAtomCustomLabel(Ot);at&&Gt.add(at);}let p1=0;for(;Gt.has(jt)&&p1++<100;)jt=Q2(jt);return jt;};}function Q2(d){let ft=d.match(/(\d+)/);if(ft){let jt=Number.parseInt(ft[1],10);return d.replace(/(\d+)/,(jt+1).toString());}let dt=d.match(/([a-yA-Y])([^a-zA-Z]*)$/);if(dt){let jt=dt[1],Gt=String.fromCodePoint(jt.codePointAt(0)+1);return Gt==="Z"||Gt==="z"?"1":d.replace(/([a-yA-Y])([^a-zA-Z]*)$/,`${Gt}$2`);}return "1";}function Yr(d,ft){switch(ft){case "superscript":for(let dt=0;dt<d.getAllAtoms();dt++){let jt=d.getAtomCustomLabel(dt);jt&&!jt.startsWith("]")&&d.setAtomCustomLabel(dt,`]${jt}`);}break;case "normal":for(let dt=0;dt<d.getAllAtoms();dt++){let jt=d.getAtomCustomLabel(dt);jt?.startsWith("]")&&d.setAtomCustomLabel(dt,jt.slice(1));}break;case "auto":for(let dt=0;dt<d.getAllAtoms();dt++){let jt=d.getAtomCustomLabel(dt);jt&&(d.getAtomLabel(dt)==="C"?jt.startsWith("]")&&d.setAtomCustomLabel(dt,jt.slice(1)):jt.startsWith("]")||d.setAtomCustomLabel(dt,`]${jt}`));}break;}}function Fu(d){let ft=d.fromMolfile;d.fromMolfile=function(jt,Gt={}){let{customLabelPosition:p1}=Gt,Ot=ft.call(this,jt),at=jt.includes(`\r
`)?`\r
`:`
`,t1=jt.split(at);if(t1.length<4||!t1[3].includes("V2000"))return Ot;let g1=t1.slice(4+Ot.getAllAtoms()+Ot.getAllBonds());for(let T1=0;T1<g1.length;T1++){let d1=g1[T1];if(d1.startsWith("A  ")){let o1=Number(d1.slice(3)),K1=g1[T1+1]?.trim();T1++,!Number.isNaN(o1)&&o1<=Ot.getAllAtoms()&&K1&&!Ot.getAtomCustomLabel(o1-1)&&Ot.setAtomCustomLabel(o1-1,K1);}if(d1.startsWith("V  ")){let o1=d1.split(" ").filter(Boolean);if(o1.length>=3){let K1=Number(o1[1]),Q0=o1.slice(2).join(" ");!Number.isNaN(K1)&&K1<=Ot.getAllAtoms()&&!Ot.getAtomCustomLabel(K1-1)&&Ot.setAtomCustomLabel(K1-1,Q0);}}if(d1.startsWith("M  ZZC")){let o1=Number(d1.slice(7,10).trim()),K1=d1.slice(10).trim();o1&&K1&&Ot.setAtomCustomLabel(o1-1,K1);}}return Yr(Ot,p1),Ot;};}var o2=["M  STY","M  SLB","M  SAL","M  SDT","M  SDD","M  SED"];function ju(d){let ft=d.prototype.toMolfile;d.prototype.toMolfile=function(jt={}){let Gt=this.getCompactCopy(),{includeCustomAtomLabelsAsALines:p1=false,includeCustomAtomLabelsAsVLines:Ot=false,customLabelPosition:at,removeCustomAtomLabels:t1=false}=jt;Yr(Gt,at);let g1=ft.call(Gt);if(!p1&&!Ot&&!t1)return g1;let T1=g1.includes(`\r
`)?`\r
`:`
`,d1=g1.split(T1);if(t1&&(d1=d1.filter(Q0=>!o2.some($1=>Q0.startsWith($1)))),d1.length<4||!d1[3].includes("V2000"))return g1;let o1=[];for(let Q0=0;Q0<Gt.getAllAtoms();Q0++){let $1=Gt.getAtomCustomLabel(Q0);if($1){let H0=String(Q0+1).padStart(3," ");p1&&o1.push(`A  ${H0}`,$1),Ot&&o1.push(`V  ${H0} ${$1}`);}}let K1=d1.findIndex(Q0=>Q0.startsWith("M  END"));return K1===-1?g1:(d1.splice(K1,0,...o1),d1.join(T1));};}function pu(d){let{ConformerGenerator:ft,ForceFieldMMFF94:dt,Molecule:jt}=d;ft.prototype.molecules=function*(){let t1;for(;(t1=this.getNextConformerAsMolecule())!==null;)yield t1;};let Gt={maxIts:4e3,gradTol:1e-4,funcTol:1e-6},p1=dt.prototype._minimise;delete dt.prototype._minimise,dt.prototype.minimise=function(t1){return t1={...Gt,...t1},p1.call(this,t1.maxIts,t1.gradTol,t1.funcTol);},Fu(jt),ju(jt),vu(jt);function Ot(at){if(!at)return null;if(at.includes("V2000")||at.includes("V3000"))return jt.fromMolfile(at);try{return jt.fromSmiles(at);}catch{}try{return jt.fromIDCode(at);}catch{}return null;}jt.fromText=function(t1){let g1=Ot(t1);return g1&&g1.getAllAtoms()>0?g1:null;},jt.prototype.getOCL=function(){return d;};}function jn(){}function l2(d){let ft=d.document,dt={},jt={userAgent:"webkit"},Gt=jn;Gt.__moduleStartupDone=jn;var t1="object",g1="anonymous",T1="fnStack",d1=`
`,o1={4:1,9:1,5:1,8:1},K1="Unknown",Q0="boolean",$1="number",H0="string",qt=2147483647,HA='For input string: "',PA="null",Ve="__noinit__",R1={4:1,5:1,8:1},m0={4:1,19:1,33:1,27:1},pn=" (copy)",yt={4:1,9:1,14:1,5:1,11:1,8:1,13:1},H1=65536,e1=65535,Z0=1e4,te="fromIndex: ",x8=" > toIndex: ",Xr=", toIndex: ",N8=", length: ",_r="Index: ",Wr=", Size: ",Ft={15:1,4:1,5:1},fn={4:1},H={7:1,4:1,5:1},WA=16777215,Y0=65280,Dn=.30000001192092896,EA=545460846592,lA=4294967296,qA=8589934592,VA=17179869184,IA=34359738368,OA=68719476736,xA=137438953472,un=274877906944,Tt={12:1,4:1,5:1},i0=536870912,$r=2.617993878,ar=3.665191429,N1=6.283185307179586,A1=3.141592653589793,v0=1.5707963267948966,N0=16384,Ie=-8355712,Z1=4096,Ae={l:0,m:0,h:1},ne={l:0,m:0,h:2},ee={l:0,m:0,h:4},X0=1920,f0=1024,mA=234881024,re=167772160,Qn=100663296,on=201326592,$A=114688,aA=4063232,Pe=3801088,Mn=3538944,Rn=3014656,ie=3145728,j0=2097152,tn=393216,iA=917504,ln=1966080,Ln=0xf8000000000,Oe=0xf0000000000,xe=0xe8000000000,Ne=0xd8000000000,Ge=0xb8000000000,tP=0xe0000000000,AP=0xc0000000000,fe=8796093022208,Cn=549755813888,ze=1649267441664,Je=3848290697216,De=8246337208320,nP=0xc8000000000,eP=9345848836096,rP=0x98000000000,vA=29360128,NA=268435456,L0=8192,B1=2048,LA=32768,ue=-1.5707963267948966,sA=130560,Qe=6291456,PP=-2.0943951023931953,c0=1.0471975511965976,Sn=.5235987755982988,l1=786432,UA=524288,yA=262144,G0=-16777216,fA=-65536,Ke=-32768,iP='<DrawingObject type="',Ze="></DrawingObject>",Yt={4:1,5:1,16:1},e0={4:1,5:1},fP="unsupported atomicNo:",G8=2.8415926535897933,E1={28:1,4:1,9:1,5:1,8:1},Hn={4:1,5:1,11:1,8:1},Y={10:1,4:1,5:1},b0={4:1,9:1,5:1,8:1,30:1},r0=131072,Ye=196608,Xe={l:6,m:0,h:4},TA=1048576,hA=3221225472,DP={l:0,m:0,h:3},sn=8388608,uP={l:0,m:0,h:120},_e={36:1,43:1,38:1},y0={4:1,9:1,5:1,11:1,8:1},z8={36:1,43:1,95:1,38:1,83:1},We={4:1,5:1,36:1,43:1,95:1,20:1,38:1,83:1,126:1},s0=17976931348623157e292,QP=-17976931348623157e292,GA={87:1,73:1,74:1,4:1,9:1,5:1,8:1},M1="??",cA={58:1,4:1,9:1,5:1,8:1},bA=2013265920,F0=32505856,E0=31744,$e=122880,En={128:1,4:1,9:1,5:1,11:1,8:1},oe=33554432,q0=67108864,ae=-131073,tr=-33554433,Ar=-33292289,oP=-3.141592653589793,W1=4194303,V0=3072,w0={46:1,4:1,5:1},lP=.7853981633974483,J8=3.061592653589793,I0=1073741824,K8=541165879296,Vt={17:1,4:1,5:1,8:1},nr=281.17,er=289.19,rr=294.21,Pr="0.0000",ir="NOSEARCH_OCL_CUSTOM_LABEL",FA="M  END",qn="ATOMS",r1=-2147483648,le="$$$$",Le="M  V30 ",Z8="M  V30 COUNTS ",Y8="M  V30 MDLV30/STEREL",X8="M  V30 MDLV30/STERAC",LP="M  V30 END CTAB",_8=34028234663852886e22,sP={l:2,m:0,h:4},_0=4194304,UP={l:4179966,m:4063288,h:7},W8={l:4194302,m:4193343,h:127},Y1='" ',yP='stroke="',fr='stroke-width="',Dr='" />',$8='class="event" ',a8='opacity="0" />',W0=16711680,ti="' position:",Ai="'. Position:",ni=1099511627776,ei=2199023255552,ri=4398046511104,TP="', position:",Pi="Assignment of aromatic double bonds failed",se=2147483646,ii={102:1,4:1},hP="Members of ESR groups must only be stereo centers with known configuration.",cP="Over- or under-specified stereo feature or more than one racemic type bond",bP="Ambiguous configuration at stereo center because of 2 parallel bonds",fi={114:1,88:1,26:1,4:1},ur=1e5,Ue=1.399999976158142,Vn=3.1415927410125732,wP=1.5707963705062866,Di=.029999999329447746,ui=6.2831854820251465,Qi=1.9106119871139526,oi=2.2639999389648438,gP=2.16759991645813,li=2.0662999153137207,Li=2.3561999797821045,si=2.2844998836517334,Ui=1.9106120066311025,yi=2.094395160675049,ye=.6000000238418579,Qr=1.7999999523162842,BP=.05000000074505806,P0={4:1,9:1,5:1,8:1,37:1},or=.2617993950843811,In=2.0943951023931953,Te=1.2000000476837158,kP=1.4299999475479126,lr=1.659999966621399,dP=1.4600000381469727,mP=1.899999976158142,Ti=1.8200000524520874,hi=2.4600000381469727,ci=2.4200000762939453,Lr=2.450000047683716,vP=2.440000057220459,bi=2.4000000953674316,FP=2.319999933242798,jP=2.2899999618530273,wi=1.8600000143051147,gi=2.5299999713897705,Bi=2.490000009536743,ki=2.4700000286102295,di=2.880000114440918,mi=2.9200000762939453,vi=2.869999885559082,pP=2.8299999237060547,Fi=2.7899999618530273,ji=2.809999942779541,pi=2.799999952316284,Mi=3.049999952316284,MP=.3199999928474426,Ri=1.3300000429153442,he=.8500000238418579,Ci=1.600000023841858,Si=1.3899999856948853,RP=1.159999966621399,Hi=1.1100000143051147,Ei=1.0700000524520874,qi=1.9600000381469727,Vi=1.7100000381469727,CP=1.3600000143051147,Ii=1.340000033378601,Oi=1.2200000286102295,xi=1.2400000095367432,sr=1.2100000381469727,Ni=2.0999999046325684,SP=1.850000023841858,HP=1.6299999952316284,Gi=1.4700000286102295,zi=1.2799999713897705,Ji=1.440000057220459,Ki=1.7599999904632568,Zi=1.7300000190734863,Yi=1.7200000286102295,Ur=1.6799999475479126,Xi=1.690000057220459,_i=1.6699999570846558,EP=1.649999976158142,ce=1.7000000476837158,Wi=1.6200000047683716,$i=1.3700000047683716,ai=1.2899999618530273,t9=1.4500000476837158,A9=1.5700000524520874,qP=-0.5235987755982988,be=-1.0471975511965976,n9=-0.7853981633974483,e9=.017453292519943295,r9=.17453292519943295,we=1e-4,Un={149:1},yn=57.29577951308232,P9=.021922173386725217,ge=1e-8,DA=2.5120960256267386,Be=1e-5,yr="MMFF94s+",i9="nonbonded cutoff",f9="dielectric constant",D9="dielectric model",u9="angle bend",Q9="bond stretch",o9="electrostatic",l9="out of plane",L9="stretch bend",s9="torsion angle",U9="van der waals",y9=.07000000000000006,T9=.1200000000000001,h9=-7.840000000000001,Tr="OCL_RXN_V1.0:",c9="$RXN V3000",b9="'$MOL' tag not found",w9="'M  END' not found",g9="M  V30 BEGIN REACTANT",B9=`
Actelion Java MolfileCreator 2.0

  0  0  0  0  0  0              0 V3000
`,k9="M  V30 BEGIN PRODUCT",d9=2475109711874,m9=2475109711875,v9=2887426572290,F9=2887426572291,j9=2887627898882,p9=2887627898883,M9=2887627997186,R9=3437182386178,C9=3437182386179,S9=3437383712770,H9=3437383811074,E9=3437652148226,q9=3437652377602,V9=4674132967426,I9=4674334294018,O9=4674334392322,x9=4674602729474,N9=4674602958850,G9=5086449827842,z9=5086449827848,J9=5086651154434,K9=5086651252738,Z9=5086651252743,Y9=5086919589890,X9=5086919688194,_9=5636406968322,W9=5636407066626,$9=5636675502082,a9={l:589826,m:590112,h:4},tf={l:589827,m:590112,h:4},Af={l:589826,m:590160,h:4},nf={l:589827,m:590160,h:4},ef={l:688130,m:590160,h:4},rf={l:589826,m:590224,h:4},Pf={l:589827,m:590224,h:4},ff={l:819203,m:590224,h:4},Df={l:589826,m:590368,h:4},uf={l:688130,m:590368,h:4},Qf={l:819202,m:590368,h:4},of={l:819208,m:590368,h:4},lf={l:589826,m:590416,h:4},Lf={l:589832,m:590416,h:4},sf={l:688130,m:590416,h:4},Uf={l:819202,m:590416,h:4},yf={l:589826,m:598304,h:4},Tf={l:589827,m:598304,h:4},hf={l:688130,m:598304,h:4},cf={l:1114114,m:598304,h:4},bf={l:1212418,m:598304,h:4},wf={l:589826,m:598308,h:4},gf={l:589826,m:688416,h:4},Bf={l:589826,m:688464,h:4},kf={l:589826,m:688720,h:4},df={l:688130,m:688720,h:4},mf={l:589826,m:590112,h:288},vf={l:589826,m:590112,h:336},Ff={l:589826,m:688416,h:336},jf={l:589826,m:688464,h:336},pf={l:688130,m:688464,h:336},Mf={l:589826,m:590112,h:400},Rf={l:589826,m:688416,h:400},Cf={l:589826,m:688464,h:400},Sf={l:688130,m:688464,h:400},Hf={l:589826,m:819488,h:400},Ef={l:589826,m:819536,h:400},qf={l:589826,m:819600,h:400},Vf={l:688130,m:819600,h:400},If={l:819202,m:819600,h:400},Of={l:589831,m:688416,h:592},xf={l:589831,m:688464,h:592},Nf={l:688135,m:688464,h:592},Gf={l:589831,m:819536,h:592},zf={l:688135,m:819536,h:592},Jf={l:688135,m:819600,h:592},Kf={l:589832,m:1212704,h:592},Zf={l:589832,m:1212752,h:592},Yf={l:688136,m:1212752,h:592},Xf={l:589831,m:688464,h:656},_f={l:589826,m:590112,h:8480},Wf={l:589826,m:688416,h:8480},$f={l:589826,m:688464,h:8480},af={l:688130,m:688464,h:8480},tD={l:819202,m:819600,h:8480},AD={l:688135,m:1212752,h:8480},nD={l:589832,m:1213008,h:8480},eD={l:688136,m:1213008,h:8480},rD={l:589826,m:590112,h:8484},PD={l:589826,m:590160,h:8484},iD={l:688130,m:598304,h:8484},fD=.6262000203132629,DD=-1.3825000524520874,hr=.33169999718666077,uD=-1.4915000200271606,QD=.3540000021457672,oD=.38179999589920044,lD=-0.6019999980926514,LD=.07999999821186066,sD=-0.7379999756813049,UD="Atom-types are 64-bit numbers describing atoms and their near surrounding.",VP="Recognized atom types and their contributions are:",yD="Druglikeness predictor not properly initialized.",TD=3.009999990463257,hD=-0.17000000178813934,cD=-0.1809999942779541,bD=-0.2029999941587448,ke=2147483648,wD="ene-Metathesis",gD="Halogenation ring closure",BD=" (rootPairSets:",IP=-1e10,cr={204:1,4:1,9:1,5:1,8:1,30:1},br=-99999,wA="any",kD="is not in a ring",OP="Excluded atoms:",wr="exactly 1",gr="exactly 2",xP="less than 3",dD="less than 4",Br="at least 1",NP="at least 2",GP="at least 3",mD="at least 4",zP="1 or 2",vD="less than 2",FD="Allowed atoms:",Tn={138:1},jD="Atom Properties",kr=-8346432,pD="/html/editor/editor.html",MD="Structure Editor Help",dr="Dummy look and feel",On="Substance Graphite",xn="Radiance Graphite",Nn="Radiance Night",RD=.6499999761581421,CD=.3499999940395355,JP="editorButtons.png",SD="esrButtons.png",hn=.800000011920929,KP=-1774190527,ZP=-1438646166,Gn=-1433403286,mr=-1431655766,vr=-1448432982,jA=1073741866,cn=1073741930,YP="MMFF94",XP="MMFF94s",HD="/resources/forcefield/mmff94/94s/outofplane.csv",zn="undefined",_P={172:1,4:1,5:1,8:1},ED=.08726646502812703,qD="overlapping",WP=.003921568859368563,M0=1048575,$0=17592186044416,bn=1e9,Jn=-17592186044416,g0=5.56,D0=11.12,pA=13.34,wn=14.44,de=.3010299956639812,VD="BigInteger divide by zero",i1=4294967295,ID={l:0,m:0,h:524288},Fr={32:1,115:1},$P={107:1,110:1},Kn={36:1,43:1,38:1,85:1},jr=16777619,OD={36:1,38:1,85:1},pr="delete",Mr=15525485,xD=5960464477539063e-23,aP=16777216,Rr={4:1,32:1,56:1,80:1},t8="Invalid UTF8 sequence",A8=.20000000298023224,ND=" distance:",GD={4:1,5:1,11:1,8:1,463:1},zD="gGP`@dfyjidNcGI[WQCP`<",Zn={48:1},o={},h={},y1={},zt={},w={},uA={},n={},U={},I={},Xt={},X={},gA={},Z={},rt={},P1={},n1={},et={},At={},a={},C={},ht={},m1={},m={},W={},R={},V1={},Lt={},j1={},B0={},Rt={},M={},q1={},o0={},l0={},z0={},mt={},_t={},_={},$={},s,me,Cr;o.xfb=function(){};function Cu(f,t){typeof window===t1&&typeof window.$gwt===t1&&(window.$gwt[f]=t);}function Su(f,t,A,e){o.kgb();var r=Cr;function P(){for(var i=0;i<r.length;i++)r[i]();}if(f)try{Au(P)();}catch(i){f(t,i);}else Au(P)();}o.kgb=function(){Cr==null&&(Cr=[]);};function Hu(){o.kgb();for(var f=Cr,t=0;t<arguments.length;t++)f.push(arguments[t]);}o.igb=function(){};function Yn(){}function Xn(f){return Array.isArray(f)&&f.yd===Yn?h.Uc(f):f.toString();}function O1(f,t){var A=d;if(f==="")return A;var e=f.split(".");if(!(e[0]in A)&&A.execScript&&A.execScript("var "+e[0]),t){var r=t.prototype.wd;r.e=t;}for(var P;e.length&&(P=e.shift());)A=A[P]=A[P]||!e.length&&t||{};return A;}o.vgb=function(t){function A(){}return A.prototype=t||{},new A();},o.ugb=function(t){return t instanceof Array?t[0]:null;};function T0(f,t,A){var e=function(){return f.apply(e,arguments);};return t.apply(e,A),e;}function f1(){}function a0(f,t){for(var A in t)t[A].configurable=true;Object.defineProperties(f,t);}function x(f,t,A){var e=me,r=e[f],P=o.ugb(r);r&&!P?s=r:(s=o.pgb(t),s.xd=A,!t&&(s.yd=Yn),e[f]=s);for(var i=3;i<arguments.length;++i)arguments[i].prototype=s;P&&(s.wd=P);}o.pgb=function(t){var A=t&&t.prototype;return !A&&(A=me[t]),o.vgb(A);},o.ogb=function(){},d.goog=d.goog||{},d.goog.global=d.goog.global||d,me={},h.Jc=function(t,A){return F1(t)===F1(A);},h.Kc=function(t){return t.wd;},h.Lc=function(t){return w.yAb(t);};function JD(){}h.Oc=function(t,A){return MA(t)?h.akb(t,A):gn(t)?h.Sib(t,A):ve(t)?h.Khb(t,A):e8(t)?t.mb(A):n8(t)?h.Jc(t,A):y1.j5(t,A);},h.Qc=function(t){return MA(t)?h.ekb():gn(t)?h.Tib():ve(t)?h.Lhb():e8(t)?t.wd:n8(t)?h.Kc(t):y1.k5(t);},h.Sc=function(t){return MA(t)?h.fkb(t):gn(t)?h.Uib(t):ve(t)?h.Mhb(t):e8(t)?t.ob():n8(t)?h.Lc(t):y1.l5(t);},h.Uc=function(t){var A;return h.mib(h.Qc(t))+"@"+(A=w.SAb(w.RAb(h.Sc(t))),A.toString(16));},x(1,null,{},JD),s.mb=function(t){return h.Jc(this,t);},s.nb=function(){return h.Kc(this);},s.ob=function(){return h.Lc(this);},s.pb=function(){var t;return h.mib(h.Qc(this))+"@"+(t=w.SAb(w.RAb(h.Sc(this))),t.toString(16));},s.equals=function(f){return this.mb(f);},s.hashCode=function(){return this.ob();},s.toString=function(){return this.pb();},y1.j5=function(t,A){return y1.p5(t)?y1.m5(t,A):F1(t)===F1(A);},y1.k5=function(t){return t.wd||Array.isArray(t)&&F(o.Rbb,1)||o.Rbb;},y1.l5=function(t){return y1.q5(t)?y1.n5(t):w.yAb(t);},y1.m5=function(t,A){return t.equals(A);},y1.n5=function(t){return t.hashCode();},y1.o5=function(){return [];},y1.p5=function(t){return !!t&&!!t.equals;},y1.q5=function(t){return !!t&&!!t.hashCode;},y1.r5=function(t){return t.toString?t.toString():"[JavaScriptObject]";},zt.X5=function(){zt.X5=f1;var t,A;A=!zt.e6(),t=new zt.n6(),zt.W5=A?new zt.f6():t;},zt.Y5=function(t){zt.X5(),zt.W5.Nc(t);},zt.Z5=function(t){zt.X5();var A;return A=zt.W5.Oc(t),zt.$5(A);},zt.$5=function(t){var A,e,r,P;for(A="Y5",e="ZB",P=d.Math.min(t.length,5),r=P-1;r>=0;r--)if(h.akb(t[r].d,A)||h.akb(t[r].d,e)){t.length>=r+1&&w.SAb(t).splice(0,r+1);break;}return t;},zt._5=function(t){var A=/function(?:\s+([\w$]+))?\s*\(/,e=A.exec(t);return e&&e[1]||g1;},zt.a6=function(t){return zt.X5(),t&&t[T1]?t[T1]:[];},zt.b6=function(t){return zt.X5(),t.name||(t.name=zt._5(t.toString()));},zt.c6=function(t){return zt.X5(),parseInt(t)||-1;},zt.d6=function(t){zt.X5();var A=t.backingJsObject;if(A&&A.stack){var e=A.stack,r=A+d1;return e.substring(0,r.length)==r&&(e=e.substring(r.length)),e.split(d1);}return [];},zt.e6=function(){return Error.stackTraceLimit>0?(d.Error.stackTraceLimit=Error.stackTraceLimit=64,true):"stack"in new Error();},x(447,1,{}),zt.f6=function(){},x(312,447,{},zt.f6),s.Nc=function(t){var A={},e=[];t[T1]=e;for(var r=arguments.callee.caller;r;){var P=zt.b6(r);e.push(P);var i=":"+P,D=A[i];if(D){var u,Q;for(u=0,Q=D.length;u<Q;u++)if(D[u]===r)return;}(D||(A[i]=[])).push(r),r=r.caller;}},s.Oc=function(t){var A,e,r,P;for(r=zt.a6(t),e=y1.x5(r),P=B(o.Kcb,o1,90,e,0,1),A=0;A<e;A++)P[A]=new h.Qjb(y1.w5(r,A),null,-1);return P;},zt.i6=function(t,A){var e,r,P,i,D,u,Q,l,L;return l="",w.SAb(A).length==0?t.Pc(K1,g1,-1,-1):(L=h.skb(A),h.akb(w.SAb(L).substr(0,3),"at ")&&(L=(w.NAb(3,w.SAb(L).length+1),w.SAb(L).substr(3))),L=zt.j6(L),D=w.SAb(L).indexOf("("),D==-1?(D=w.SAb(L).indexOf("@"),D==-1?(l=L,L=""):(l=h.skb((w.NAb(D+1,w.SAb(L).length+1),w.SAb(L).substr(D+1))),L=h.skb((w.MAb(0,D,w.SAb(L).length),w.SAb(L).substr(0,D))))):(e=w.SAb(L).indexOf(")",D),l=(w.MAb(D+1,e,w.SAb(L).length),w.SAb(L).substr(D+1,e-(D+1))),L=h.skb((w.MAb(0,D,w.SAb(L).length),w.SAb(L).substr(0,D)))),D=h.hkb(L,X1(46)),D!=-1&&(L=(w.NAb(D+1,w.SAb(L).length+1),w.SAb(L).substr(D+1))),(w.SAb(L).length==0||h.akb(L,"Anonymous function"))&&(L=g1),u=h.kkb(l,X1(58)),P=h.lkb(l,X1(58),u-1),Q=-1,r=-1,i=K1,u!=-1&&P!=-1&&(i=(w.MAb(0,P,w.SAb(l).length),w.SAb(l).substr(0,P)),Q=zt.c6((w.MAb(P+1,u,w.SAb(l).length),w.SAb(l).substr(P+1,u-(P+1)))),r=zt.c6((w.NAb(u+1,w.SAb(l).length+1),w.SAb(l).substr(u+1)))),t.Pc(i,L,Q,r));},zt.j6=function(t){return t.replace(/\[.*?\]/g,"");},x(448,447,{}),s.Nc=function(t){},s.Pc=function(t,A,e,r){return new h.Qjb(A,t+"@"+r,e<0?-1:e);},s.Oc=function(t){var A,e,r,P,i,D;if(P=zt.d6(t),i=B(o.Kcb,o1,90,0,0,1),A=0,r=y1.x5(P),r==0)return i;for(D=zt.i6(this,y1.w5(P,0)),h.akb(D.d,g1)||(i[A++]=D),e=1;e<r;e++)i[A++]=zt.i6(this,y1.w5(P,e));return i;},zt.n6=function(){},x(313,448,{},zt.n6),s.Pc=function(t,A,e,r){return new h.Qjb(A,t,-1);},o.p6=function(t){return t;};function F(f,t){return o.r6(f,t);}o.r6=function(t,A){return h.xib(t,A);},o.s6=function(t){return t.__elementTypeCategory$==null?10:t.__elementTypeCategory$;},o.t6=function(t){return t.__elementTypeId$;};function U1(f,t,A,e,r,P){return o.v6(f,t,A,e,r,0,P);}o.v6=function(t,A,e,r,P,i,D){var u,Q,l,L,y;if(L=P[i],l=i==D-1,u=l?r:0,y=o.x6(u,L),r!=10&&j(F(t,D-i),A[i],e[i],u,y),!l)for(++i,Q=0;Q<L;++Q)o.z6(y,Q,o.v6(t,A,e,r,P,i,D));return y;};function B(f,t,A,e,r,P){var i;return i=o.x6(r,e),r!=10&&j(F(f,P),t,A,r,i),i;}o.x6=function(t,A){var e=new Array(A),r;switch(t){case 14:case 15:r=0;break;case 16:r=false;break;default:return e;}for(var P=0;P<A;++P)e[P]=r;return e;};function n8(f){return Array.isArray(f)&&f.yd===Yn;}o.z6=function(t,A,e){return t[A]=e;};function a1(f,t,A){return o.z6(f,t,A);}o.B6=function(t,A){t.wd=A;},o.C6=function(t,A){t.__elementTypeCategory$=A;},o.D6=function(t,A){t.__elementTypeId$=A;};function j(f,t,A,e,r){return o.B6(r,f),r.xd=t,r.yd=Yn,o.D6(r,A),o.C6(r,e),r;}o.F6=function(t,A){return o.s6(A)!=10&&j(h.Qc(A),A.xd,o.t6(A),o.s6(A),t),o.p6(t);};function Eu(f,t){return MA(f)?!!_D[t]:f.xd?!!f.xd[t]:gn(f)?!!XD[t]:ve(f)?!!YD[t]:false;}function e8(f){return !Array.isArray(f)&&f.yd===Yn;}function Zt(f,t){return f!=null&&Eu(f,t);}function ve(f){return typeof f===Q0;}function gn(f){return typeof f===$1;}function KD(f){return f!=null&&o.I7(f)&&f.yd!==Yn;}function ZD(f,t){return o.J7(f,t);}function MA(f){return typeof f===H0;}o.I7=function(t){return typeof t===t1||typeof t=="function";},o.J7=function(t,A){return t&&A&&t instanceof A;};function F1(f){return f??null;}function Mt(f){return Math.max(Math.min(f,qt),-2147483648)|0;}var YD,XD,_D;o.yfb=function(t){return t&&t.__java$exception;};function C1(f){var t;return Zt(f,27)?f:(t=o.yfb(f),t||(t=new y1.d5(f),zt.Y5(t)),t);}function st(f){return f.backingJsObject;}function qu(f){var t;return f==null?false:(t=typeof f,h.akb(t,Q0)||h.akb(t,$1)||h.akb(t,H0)||f.$implements__java_io_Serializable||Array.isArray(f));}function Vu(f){var t;return t=typeof f,h.akb(t,Q0)||h.akb(t,$1)||h.akb(t,H0)?true:f!=null&&f.$implements__java_lang_Comparable;}h.Ghb=function(){h.Ghb=f1,Fe=w.SAb(false),w.SAb(true);},h.Hhb=function(t){return w.TAb((w.GAb(t),t));},h.Ihb=function(t,A){return Ou(w.TAb((w.GAb(t),t)),w.TAb((w.GAb(A),A)));},h.Jhb=function(t,A){return h.Ihb(t,A);},h.Khb=function(t,A){return w.GAb(t),F1(t)===F1(A);},h.Lhb=function(){return o.scb;},h.Mhb=function(t){return w.TAb((w.GAb(t),t))?1231:1237;};function Iu(f){return h.Ghb(),h.akb(Q0,typeof f);}function Ou(f,t){return h.Ghb(),f==t?0:f?1:-1;}h.Phb=function(t,A){return h.Ghb(),MA(t)?h.Vjb(t,A):gn(t)?h.Qib(t,A):ve(t)?h.Jhb(t,A):t.Sb(A);},YD={4:1,307:1,32:1};var Fe;function xu(f){return h.akb(typeof f,H0)?true:f!=null&&f.$implements__java_lang_CharSequence;}h.kib=function(t){t.i=zu++;},h.lib=function(t){t.n==null&&h.Bib(t);},h.mib=function(t){return h.lib(t),t.n;},h.nib=function(t){return (t.f&4)!=0;},h.oib=function(t){return (t.f&1)!=0;},h.pib=function(){h.kib(this),this.n=null,this.j=null,this.g=null,this.d=null,this.b=null,this.k=null,this.a=null;},h.rib=function(t){var A;return A=new h.pib(),A.n="Class$"+(t?"S"+t:""+A.i),A.b=A.n,A.j=A.n,A;};function N(f){var t;return t=h.rib(f),h.Fib(f,t),t;}function zA(f,t){var A;return A=h.rib(f),h.Fib(f,A),A.f=t?8:0,A;}function Nu(){var f;return f=h.rib(null),f.f=2,f;}function An(f){var t;return t=h.rib(f),t.k=f,t.f=1,t;}h.xib=function(t,A){var e=t.a=t.a||[];return e[A]||(e[A]=t.Vc(A));};function Gu(f){if(f.$c())return null;var t=f.k;return me[t];}h.Bib=function(t){if(t.Zc()){var A=t.c;A.$c()?t.n="["+A.k:A.Zc()?t.n="["+A.Xc():t.n="[L"+A.Xc()+";",t.b=A.Wc()+"[]",t.j=A.Yc()+"[]";return;}var e=t.g,r=t.d;r=r.split("/"),t.n=h.Eib(".",[e,h.Eib("$",r)]),t.b=h.Eib(".",[e,h.Eib(".",r)]),t.j=r[r.length-1];},h.Eib=function(t,A){for(var e=0;!A[e]||A[e]=="";)e++;for(var r=A[e++];e<A.length;e++)!A[e]||A[e]==""||(r+=t+A[e]);return r;},h.Fib=function(t,A){if(t){A.k=t;var r=Gu(A);if(!r){me[t]=[A];return;}r.wd=A;}},x(207,1,{},h.pib),s.Vc=function(t){var A;return A=new h.pib(),A.f=4,t>1?A.c=h.xib(this,t-1):A.c=this,A;},s.Wc=function(){return h.lib(this),this.b;},s.Xc=function(){return h.mib(this);},s.Yc=function(){return h.lib(this),this.j;},s.Zc=function(){return h.nib(this);},s.$c=function(){return h.oib(this);},s.pb=function(){return ((this.f&2)!=0?"interface ":(this.f&1)!=0?"":"class ")+(h.lib(this),this.n);},s.f=0,s.i=0;var zu=1;h.Kib=function(t){return h.akb($1,typeof t)||ZD(t,d.java.lang.Number$impl);},h.Lib=function(t){if(h.Jib==null&&(h.Jib=new RegExp("^\\s*[+-]?(NaN|Infinity|((\\d+\\.?\\d*)|(\\.\\d+))([eE][+-]?\\d+)?[dDfF]?)\\s*$")),!h.Jib.test(t))throw st(new h.Pjb(HA+t+'"'));return parseFloat(t);},h.Mib=function(t,A,e){var r,P,i,D,u;if(t==null)throw st(new h.Pjb(PA));for(i=w.SAb(t).length,D=i>0&&(w.NAb(0,w.SAb(t).length),w.SAb(t).charCodeAt(0)==45||(w.NAb(0,w.SAb(t).length),w.SAb(t).charCodeAt(0)==43))?1:0,r=D;r<i;r++)if($u((w.NAb(r,w.SAb(t).length),w.SAb(t).charCodeAt(r)))==-1)throw st(new h.Pjb(HA+t+'"'));if(u=parseInt(t,10),P=u<A,isNaN(u))throw st(new h.Pjb(HA+t+'"'));if(P||u>e)throw st(new h.Pjb(HA+t+'"'));return u;},h.Nib=function(t){return gn(t)?h.Rib(t):t._c();},h.Oib=function(t){return gn(t)?h.Vib(t):t.ad();},x(120,1,{4:1,120:1}),h.Pib=function(t,A){return JA(w.UAb((w.GAb(t),t)),w.UAb((w.GAb(A),A)));},h.Qib=function(t,A){return h.Pib(t,A);},h.Rib=function(t){return w.UAb((w.GAb(t),t));},h.Sib=function(t,A){return w.GAb(t),F1(t)===F1(A);},h.Tib=function(){return o.vcb;},h.Uib=function(t){return Mt(w.UAb((w.GAb(t),t)));},h.Vib=function(t){return Mt(w.UAb((w.GAb(t),t)));};function Ju(f){return h.akb($1,typeof f);}function JA(f,t){return f<t?-1:f>t?1:f==t?f==0?JA(1/f,1/t):0:isNaN(f)?isNaN(t)?0:1:-1;}function Sr(f){return isNaN(f)?{l:0,m:0,h:524160}:w.OAb(f);}XD={4:1,32:1,308:1,120:1},h.GB=function(t){t.g=B(o.Kcb,o1,90,0,0,1);},h.HB=function(t){zt.Y5(t);},h.IB=function(t){return zt.Z5(t);},h.JB=function(t){return t.j&&(t.backingJsObject!==Ve&&t.jc(),t.g=null),t;},h.KB=function(t){return t.e;},h.LB=function(t,A){if(A instanceof Object)try{if(A.__java$exception=t,jt.userAgent.toLowerCase().indexOf("msie")!=-1&&ft.documentMode<9)return;var e=t;Object.defineProperties(A,{cause:{get:function(){var r=e.hc();return r&&r.fc();}},suppressed:{get:function(){return e.gc();}}});}catch{}},h.MB=function(t,A,e){var r,P,i,D,u;for(h.NB(t),P=(t.i==null&&(t.i=B(o.Pcb,o1,27,0,0,1)),t.i),i=0,D=P.length;i<D;++i)r=P[i],h.MB(r,A,"	"+e);u=t.e,u&&h.MB(u,A,e);},h.NB=function(t){var A,e,r;for(A=(t.g==null&&(t.g=h.IB(t)),t.g),e=0,r=A.length;e<r;++e);},h.OB=function(t,A){t.backingJsObject=A,h.LB(t,A);},h.PB=function(t){return h.QB(t,t.ic());},h.QB=function(t,A){var e;return e=h.mib(t.wd),A==null?e:e+": "+A;},h.RB=function(){h.GB(this),h.JB(this),this.jc();},h.SB=function(t){h.GB(this),this.f=t,h.JB(this),this.jc();},h.UB=function(t){if(!("stack"in t))try{throw t;}catch{}return t;};function Ku(f){var t;return f!=null&&(t=f.__java$exception,t)?t:ZD(f,TypeError)?new h.Mjb(f):new h.X4(f);}x(27,1,{4:1,27:1}),s.ec=function(t){return new Error(t);},s.fc=function(){return this.backingJsObject;},s.gc=function(){var t,A,e;for(e=(this.i==null&&(this.i=B(o.Pcb,o1,27,0,0,1)),this.i),A=B(o.Icb,R1,1,e.length,5,1),t=0;t<e.length;t++)A[t]=e[t].backingJsObject;return A;},s.hc=function(){return h.KB(this);},s.ic=function(){return this.f;},s.jc=function(){h.OB(this,h.UB(this.ec(h.QB(this,this.f)))),h.HB(this);},s.pb=function(){return h.PB(this);},s.backingJsObject=Ve,s.j=true,h.aC=function(){h.RB.call(this);},h.bC=function(t){h.GB(this),h.JB(this),this.backingJsObject=t,h.LB(this,t),this.f=t==null?PA:Xn(t);},h.cC=function(t){h.SB.call(this,t);},x(19,27,{4:1,19:1,27:1},h.cC),h.dC=function(){h.aC.call(this);},h.eC=function(t){h.bC.call(this,t);},h.fC=function(t){h.cC.call(this,t);},x(33,19,m0),h.Bhb=function(){h.dC.call(this);},h.Chb=function(t){h.fC.call(this,t);},x(79,33,m0,h.Bhb,h.Chb),h.W4=function(){h.dC.call(this);},h.X4=function(t){h.eC.call(this,t);},h.Y4=function(t){h.fC.call(this,t);},x(129,33,m0,h.X4),h.Ljb=function(){h.W4.call(this);},h.Mjb=function(t){h.X4.call(this,t);},h.Njb=function(t){h.Y4.call(this,t);},x(69,129,m0,h.Ljb,h.Mjb,h.Njb),s.ec=function(t){return new TypeError(t);},h.Ujb=function(t,A){return w.NAb(A,w.SAb(t).length),w.SAb(t).charCodeAt(A);},h.Vjb=function(t,A){return h.Wjb(t,A);},h.Wjb=function(t,A){var e,r;return e=h.Rib(w.SAb(t)),r=h.Rib(w.SAb(A)),e==r?0:e<r?-1:1;},h.Xjb=function(t,A){return h.Wjb(w.SAb(t).toLowerCase(),w.SAb(A).toLowerCase());},h.Yjb=function(t,A){return w.GAb(t),t+(""+(w.GAb(A),A));},h.Zjb=function(t){var A;return h.xkb(w.qAb(t,0,(A=t.length,w.nAb(),A)));},h.$jb=function(t){return h.xkb(w.qAb(t,0,t.length));},h._jb=function(t){var A;return A=w.SAb(pn).length,h.akb(w.SAb(t).substr(w.SAb(t).length-A,A),pn);},h.akb=function(t,A){return w.GAb(t),F1(t)===F1(A);},h.bkb=function(t,A){return w.GAb(t),A==null?false:h.akb(t,A)?true:w.SAb(t).length==w.SAb(A).length&&h.akb(w.SAb(t).toLowerCase(),w.SAb(A).toLowerCase());},h.ckb=function(t,A,e,r,P){w.MAb(A,e,w.SAb(t).length),w.MAb(P,P+(e-A),r.length),h.dkb(t,A,e,r,P);},h.dkb=function(t,A,e,r,P){for(;A<e;)r[P++]=h.Ujb(t,A++);},h.ekb=function(){return o.Ocb;},h.fkb=function(t){var A,e;for(A=0,e=0;e<w.SAb(t).length;e++)A=(A<<5)-A+(w.NAb(e,w.SAb(t).length),w.SAb(t).charCodeAt(e))|0;return A;},h.gkb=function(t,A,e){return h.ikb(t,X1(A),e);},h.hkb=function(t,A){return w.SAb(t).indexOf(A);},h.ikb=function(t,A,e){return w.SAb(t).indexOf(A,e);};function Zu(f){return h.akb(H0,typeof f);}h.kkb=function(t,A){return w.SAb(t).lastIndexOf(A);},h.lkb=function(t,A,e){return w.SAb(t).lastIndexOf(A,e);},h.mkb=function(t,A){var e;return e=h.nkb(A,"([/\\\\\\.\\*\\+\\?\\|\\(\\)\\[\\]\\{\\}$^])","\\\\$1"),h.nkb(t,e,"");},h.nkb=function(t,A,e){return e=h.wkb(e),w.SAb(t).replace(new RegExp(A,"g"),e);},h.okb=function(t,A){var e,r,P,i,D,u,Q,l;for(e=new RegExp(A,"g"),Q=B(o.Ocb,yt,2,0,6,1),r=0,l=t,i=null;;)if(u=e.exec(l),u==null||l==""){Q[r]=l;break;}else D=u.index,Q[r]=(w.MAb(0,D,w.SAb(l).length),w.SAb(l).substr(0,D)),l=h.rkb(l,D+w.SAb(w.SAb(u)[0]).length,w.SAb(l).length),e.lastIndex=0,i==l&&(Q[r]=(w.MAb(0,1,w.SAb(l).length),w.SAb(l).substr(0,1)),l=(w.NAb(1,w.SAb(l).length+1),w.SAb(l).substr(1))),i=l,++r;if(w.SAb(t).length>0){for(P=Q.length;P>0&&Q[P-1]=="";)--P;P<Q.length&&(w.SAb(Q).length=P);}return Q;},h.pkb=function(t,A){return h.akb(w.SAb(t).substr(0,w.SAb(A).length),A);},h.qkb=function(t,A){return w.NAb(A,w.SAb(t).length+1),w.SAb(t).substr(A);},h.rkb=function(t,A,e){return w.MAb(A,e,w.SAb(t).length),w.SAb(t).substr(A,e-A);},h.skb=function(t){var A,e,r;for(e=w.SAb(t).length,r=0;r<e&&(w.NAb(r,w.SAb(t).length),w.SAb(t).charCodeAt(r)<=32);)++r;for(A=e;A>r&&(w.NAb(A-1,w.SAb(t).length),w.SAb(t).charCodeAt(A-1)<=32);)--A;return r>0||A<e?(w.MAb(r,A,w.SAb(t).length),w.SAb(t).substr(r,A-r)):t;},h.tkb=function(t){return String.fromCharCode.apply(null,t);};function X1(f){var t,A;return f>=H1?(t=55296+(f-H1>>10&1023)&e1,A=56320+(f-H1&1023)&e1,String.fromCharCode(t)+(""+String.fromCharCode(A))):String.fromCharCode(f&e1);}h.vkb=function(t){var A,e,r,P,i;for(i=new U.Ewb(),e=t,r=0,P=e.length;r<P;++r)A=e[r],i.a?h.Ikb(i.a,i.b):i.a=new h.Mkb(i.d),h.Hkb(i.a,A);return i.a?w.SAb(i.e).length==0?i.a.a:i.a.a+(""+i.e):i.c;},h.wkb=function(t){var A;for(A=0;0<=(A=w.SAb(t).indexOf("\\",A));)w.NAb(A+1,w.SAb(t).length),w.SAb(t).charCodeAt(A+1)==36?t=(w.MAb(0,A,w.SAb(t).length),w.SAb(t).substr(0,A)+"$"+h.qkb(t,++A)):t=(w.MAb(0,A,w.SAb(t).length),w.SAb(t).substr(0,A)+(""+h.qkb(t,++A)));return t;},h.xkb=function(t){return h.ykb(t,0,t.length);},h.ykb=function(t,A,e){var r,P,i,D;for(i=A+e,w.MAb(A,i,t.length),D="",P=A;P<i;)r=d.Math.min(P+Z0,i),D+=h.tkb(w.SAb(t).slice(P,r)),P=r;return D;},_D={4:1,206:1,32:1,2:1},h.Nkb=function(t){h.Chb.call(this,t);},x(231,79,m0,h.Nkb),w.cAb=function(t){var A;return A=w.SAb(t).slice(),w.mAb(A,t);},w.dAb=function(t,A,e,r,P,i){var D,u,Q,l,L;if(P!=0)for(F1(t)===F1(e)&&(t=w.SAb(t).slice(A,A+P),A=0),Q=w.SAb(e),u=A,l=A+P;u<l;)D=d.Math.min(u+Z0,l),P=D-u,L=w.SAb(t).slice(u,D),w.SAb(L).splice(0,0,w.SAb(r),w.SAb(i?P:0)),Array.prototype.splice.apply(Q,L),u=D,r+=P;},w.eAb=function(t,A){return w.mAb(new Array(A),t);},w.fAb=function(t,A,e){w.SAb(t).splice(A,0,e);},w.gAb=function(t,A,e){w.dAb(e,0,t,A,e.length,false);},w.hAb=function(t,A){w.SAb(t).push(A);},w.iAb=function(t,A){w.SAb(t).push(A);},w.jAb=function(t,A,e){w.SAb(t).splice(A,e);},w.kAb=function(t,A){w.SAb(t).length=A;},w.lAb=function(t,A){w.SAb(t).sort(A);},x(513,1,{}),w.vAb=function(){},w.wAb=function(t){switch(typeof t){case H0:return h.fkb(w.SAb(t));case $1:return h.Uib(w.SAb(t));case Q0:return h.Mhb(w.SAb(t));default:return t==null?0:w.yAb(t);}},w.xAb=function(){return ++w.uAb;},w.yAb=function(t){return t.$H||(t.$H=w.xAb());},x(445,1,{},w.vAb),w.uAb=0,w.zAb=function(t){if(!t)throw st(new h.$ib());},w.AAb=function(t,A){if(!t)throw st(new h._ib(A));},w.BAb=function(t,A,e){if(t>A)throw st(new h._ib(te+t+x8+A));if(t<0||A>e)throw st(new h.Ehb(te+t+Xr+A+N8+e));},w.CAb=function(t){if(t<0)throw st(new h.Kjb("Negative array size: "+t));},w.DAb=function(t,A){if(t!=A)throw st(new U.mub());},w.EAb=function(t){if(!t)throw st(new U.Gvb());},w.FAb=function(t,A){if(t<0||t>=A)throw st(new h.Chb(_r+t+Wr+A));},w.GAb=function(t){if(t==null)throw st(new h.Ljb());return t;},w.HAb=function(t,A){if(t==null)throw st(new h.Njb(A));},w.IAb=function(t,A){if(t<0||t>A)throw st(new h.Chb(_r+t+Wr+A));},w.JAb=function(t,A,e){if(t<0||A>e)throw st(new h.Chb(te+t+Xr+A+", size: "+e));if(t>A)throw st(new h._ib(te+t+x8+A));},w.KAb=function(t){if(!t)throw st(new h.ajb());},w.LAb=function(t){if(!t)throw st(new h.bjb("Unable to add element to queue"));},w.MAb=function(t,A,e){if(t<0||A>e||A<t)throw st(new h.Nkb(te+t+Xr+A+N8+e));},w.NAb=function(t,A){if(t<0||t>=A)throw st(new h.Nkb(_r+t+Wr+A));},w.OAb=function(t){var A,e;return A=new ArrayBuffer(8),w.SAb(new Float64Array(A))[0]=t,e=w.SAb(new Uint32Array(A)),w.VAb(e[0]|0,e[1]|0);},w.PAb=function(t,A){return t[A];},w.QAb=function(t){return t===void 0;},w.RAb=function(t){return t>>>0;},w.SAb=function(t){return t;},w.TAb=function(t){return t;},w.UAb=function(t){return t;},x(446,1,{}),o.Icb=N(1),o.Rbb=N(0),o.Ybb=N(447),o.Vbb=N(312),o.Xbb=N(448),o.Wbb=N(313),o.scb=N(307),o.ucb=N(207),o.Hcb=N(120),o.vcb=N(308),o.Pcb=N(27),o.xcb=N(19),o.Jcb=N(33),o.Acb=N(79),o.Ccb=N(129),o.Fcb=N(69),o.Ocb=N(2),o.Ncb=N(231),o.Peb=N(445),o.Qeb=N(446),uA.Vc=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It;for(this.c=t.length,this.d=t[0].length,z=d.Math.min(this.c,this.d),this.e=B(o.O7,Ft,6,d.Math.min(this.c+1,this.d),15,1),this.a=U1(o.O7,[o1,Ft],[15,6],15,[this.c,z],2),this.b=U1(o.O7,[o1,Ft],[15,6],15,[this.d,this.d],2),P=B(o.O7,Ft,6,this.d,15,1),It=B(o.O7,Ft,6,this.c,15,1),tt=d.Math.min(this.c-1,this.d),Qt=d.Math.max(0,d.Math.min(this.d-2,this.c)),q=0;q<d.Math.max(tt,Qt);q++){if(q<tt){for(this.e[q]=0,y=q;y<this.c;y++)this.e[q]=uA.Wc(this.e[q],t[y][q]);if(this.e[q]!=0){for(t[q][q]<0&&(this.e[q]=-this.e[q]),L=q;L<this.c;L++)t[L][q]/=this.e[q];t[q][q]+=1;}this.e[q]=-this.e[q];}for(k=q+1;k<this.d;k++){if(q<tt&this.e[q]!=0){for(ct=0,y=q;y<this.c;y++)ct+=t[y][q]*t[y][k];for(ct=-ct/t[q][q],L=q;L<this.c;L++)t[L][k]+=ct*t[L][q];}P[k]=t[q][k];}if(true&q<tt)for(L=q;L<this.c;L++)this.a[L][q]=t[L][q];if(q<Qt){for(P[q]=0,y=q+1;y<this.d;y++)P[q]=uA.Wc(P[q],P[y]);if(P[q]!=0){for(P[q+1]<0&&(P[q]=-P[q]),T=q+1;T<this.d;T++)P[T]/=P[q];P[q+1]+=1;}if(P[q]=-P[q],q+1<this.c&P[q]!=0){for(T=q+1;T<this.c;T++)It[T]=0;for(v=q+1;v<this.d;v++)for(c=q+1;c<this.c;c++)It[c]+=P[v]*t[c][v];for(S=q+1;S<this.d;S++)for(ct=-P[S]/P[q+1],c=q+1;c<this.c;c++)t[c][S]+=ct*It[c];}for(L=q+1;L<this.d;L++)this.b[L][q]=P[L];}}for(it=d.Math.min(this.d,this.c+1),tt<this.d&&(this.e[tt]=t[tt][tt]),this.c<it&&(this.e[it-1]=0),Qt+1<it&&(P[Qt]=t[Qt][it-1]),P[it-1]=0,p=tt;p<z;p++){for(L=0;L<this.c;L++)this.a[L][p]=0;this.a[p][p]=1;}for(V=tt-1;V>=0;V--)if(this.e[V]!=0){for(g=V+1;g<z;g++){for(ct=0,y=V;y<this.c;y++)ct+=this.a[y][V]*this.a[y][g];for(ct=-ct/this.a[V][V],T=V;T<this.c;T++)this.a[T][g]+=ct*this.a[T][V];}for(c=V;c<this.c;c++)this.a[c][V]=-this.a[c][V];for(this.a[V][V]=1+this.a[V][V],L=0;L<V-1;L++)this.a[L][V]=0;}else {for(L=0;L<this.c;L++)this.a[L][V]=0;this.a[V][V]=1;}for(G=this.d-1;G>=0;G--){if(G<Qt&P[G]!=0)for(g=G+1;g<z;g++){for(ct=0,y=G+1;y<this.d;y++)ct+=this.b[y][G]*this.b[y][g];for(ct=-ct/this.b[G+1][G],T=G+1;T<this.d;T++)this.b[T][g]+=ct*this.b[T][G];}for(L=0;L<this.d;L++)this.b[L][G]=0;this.b[G][G]=1;}for(lt=it-1,u=d.Math.pow(2,-52);it>0;){for(E=it-2;E>=-1&&E!=-1;E--)if(d.Math.abs(P[E])<=u*(d.Math.abs(this.e[E])+d.Math.abs(this.e[E+1]))){P[E]=0;break;}if(E==it-2)K=4;else {for(O=it-1;O>=E&&O!=E;O--)if(ct=(O!=it?d.Math.abs(P[O]):0)+(O!=E+1?d.Math.abs(P[O-1]):0),d.Math.abs(this.e[O])<=u*ct){this.e[O]=0;break;}O==E?K=3:O==it-1?K=1:(K=2,E=O);}switch(++E,K){case 1:for(Q=P[it-2],P[it-2]=0,g=it-2;g>=E;g--)for(ct=uA.Wc(this.e[g],Q),r=this.e[g]/ct,bt=Q/ct,this.e[g]=ct,g!=E&&(Q=-bt*P[g-1],P[g-1]=r*P[g-1]),L=0;L<this.d;L++)ct=r*this.b[L][g]+bt*this.b[L][it-1],this.b[L][it-1]=-bt*this.b[L][g]+r*this.b[L][it-1],this.b[L][g]=ct;break;case 2:for(Q=P[E-1],P[E-1]=0,g=E;g<it;g++)for(ct=uA.Wc(this.e[g],Q),r=this.e[g]/ct,bt=Q/ct,this.e[g]=ct,Q=-bt*P[g],P[g]=r*P[g],L=0;L<this.c;L++)ct=r*this.a[L][g]+bt*this.a[L][E-1],this.a[L][E-1]=-bt*this.a[L][g]+r*this.a[L][E-1],this.a[L][g]=ct;break;case 3:{for(ut=d.Math.max(d.Math.max(d.Math.max(d.Math.max(d.Math.abs(this.e[it-1]),d.Math.abs(this.e[it-2])),d.Math.abs(P[it-2])),d.Math.abs(this.e[E])),d.Math.abs(P[E])),Ut=this.e[it-1]/ut,wt=this.e[it-2]/ut,D=P[it-2]/ut,ot=this.e[E]/ut,i=P[E]/ut,A=((wt+Ut)*(wt-Ut)+D*D)/2,e=Ut*D*(Ut*D),Dt=0,A!=0|e!=0&&(Dt=d.Math.sqrt(A*A+e),A<0&&(Dt=-Dt),Dt=e/(A+Dt)),Q=(ot+Ut)*(ot-Ut)+Dt,l=ot*i,g=E;g<it-1;g++){for(ct=uA.Wc(Q,l),r=Q/ct,bt=l/ct,g!=E&&(P[g-1]=ct),Q=r*this.e[g]+bt*P[g],P[g]=r*P[g]-bt*this.e[g],l=bt*this.e[g+1],this.e[g+1]=r*this.e[g+1],y=0;y<this.d;y++)ct=r*this.b[y][g]+bt*this.b[y][g+1],this.b[y][g+1]=-bt*this.b[y][g]+r*this.b[y][g+1],this.b[y][g]=ct;if(ct=uA.Wc(Q,l),r=Q/ct,bt=l/ct,this.e[g]=ct,Q=r*P[g]+bt*this.e[g+1],this.e[g+1]=-bt*P[g]+r*this.e[g+1],l=bt*P[g+1],P[g+1]=r*P[g+1],g<this.c-1)for(L=0;L<this.c;L++)ct=r*this.a[L][g]+bt*this.a[L][g+1],this.a[L][g+1]=-bt*this.a[L][g]+r*this.a[L][g+1],this.a[L][g]=ct;}P[it-2]=Q;}break;case 4:{if(this.e[E]<=0)for(this.e[E]=this.e[E]<0?-this.e[E]:0,L=0;L<=lt;L++)this.b[L][E]=-this.b[L][E];for(;E<lt&&!(this.e[E]>=this.e[E+1]);){if(ct=this.e[E],this.e[E]=this.e[E+1],this.e[E+1]=ct,E<this.d-1)for(L=0;L<this.d;L++)ct=this.b[L][E+1],this.b[L][E+1]=this.b[L][E],this.b[L][E]=ct;if(E<this.c-1)for(L=0;L<this.c;L++)ct=this.a[L][E+1],this.a[L][E+1]=this.a[L][E],this.a[L][E]=ct;++E;}--it;}}}},uA.Wc=function(t,A){var e;return d.Math.abs(t)>d.Math.abs(A)?(e=A/t,e=d.Math.abs(t)*d.Math.sqrt(1+e*e)):A!=0?(e=t/A,e=d.Math.abs(A)*d.Math.sqrt(1+e*e)):e=0,e;},x(200,1,fn,uA.Vc),s.c=0,s.d=0,o.S7=N(200),n._c=function(){n._c=f1,n.Xc=j(F(o.Q7,1),H,6,15,[0,WA,14286847,13402367,12779264,16758197,9474192,3166456,16715021,9494608,11789301,11230450,9109248,12560038,1578e4,16744448,16777008,2093087,8442339,9388244,4062976,15132390,12567239,10921643,9083335,10255047,14706227,15765664,5296208,13140019,8224944,12750735,6721423,12419299,16752896,10889513,6076625,7351984,Y0,9764863,9756896,7586505,5551541,3907230,2396047,687500,27013,12632256,16767375,10909043,6717568,10380213,13924864,9699476,4366e3,5707663,51456,7394559,16777159,14286791,13107143,10747847,9437127,6422471,4587463,3211207,2097095,65436,58997,54354,48952,43812,5096191,5089023,2200790,2522539,2516630,1528967,13684960,16765219,12105936,10900557,5724513,10375093,11230208,7688005,4358806,4325478,32e3,7384058,47871,41471,36863,33023,27647,5528818,7888099,9064419,10565332,11739092,11739066,11734438,12389767,13041766,13369433,13697103,14221381,14680120,15073326,15400998,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13158600,1334015,56540,15075850,15132160,56540,15075850,15461355,8553170,1016335,1016335,1334015,15132160,3289770,14456450,16422400,16422400,11819700,3289770,1016335]);},n.ad=function(t){t.v=new ht.oO();},n.bd=function(t){var A,e;(t.G&32)==0&&(e=n.ew(t.L),e!=null&&(t.w.a==0&&t.w.b==0&&(A=t.R.c*n.Ul(t.L),t.v=n.Zd(t),n.od(t,A),n.Ud(t,null,A,0)),t.Cb(Mt(t.A)),t.L.O!=1&&n.Vd(t,448),t.tb(e,t.w.a,t.w.b+Dn*t.A)));},n.cd=function(t,A){return t==null?A:A==null?t:t+","+A;},n.dd=function(t,A){n.ui(A,t.R),n.xi(A,t.v),n.wi(A,t.w);},n.ed=function(t){var A;A=t.R.c*(t.r!=0?t.r:n.Ul(t.L)),t.Y=A*n.Zc,t.T=A*n.Yc,t.S=A*.38,t.W=A*.47,t.X=Mt(A*t.J*n.$c+.5),t.V=A*.12,t.Z=A*.4,t.A=A*.5+.5;},n.fd=function(t){var A,e;for(t=J(t,EA),e=0;e<(C.sK(),C.rK).length;e++)if(S1(t,C.rK[e]))return C.qK[e];return A=new h.Mkb("R"),pt(J(t,lA),0)&&(A.a+="0"),pt(J(t,qA),0)&&(A.a+="3"),pt(J(t,VA),0)&&(A.a+="4"),pt(J(t,IA),0)&&(A.a+="5"),pt(J(t,OA),0)&&(A.a+="6"),pt(J(t,xA),0)&&(A.a+="7"),pt(J(t,un),0)&&(A.a+="8"),A.a;},n.gd=function(t,A,e,r){var P,i;P=new n.je(),i=new n.je(),P.a=A.a,P.c=A.c,P.b=(A.a+A.b)/2,P.d=(A.c+A.d)/2,i.a=P.b,i.c=P.d,i.b=A.b,i.d=A.d,n.Rd(t,P)&&(n.Vd(t,t.p[e]),t.qb(P)),n.Rd(t,i)&&(n.Vd(t,t.p[r]),t.qb(i)),n.Vd(t,t.Q);},n.hd=function(t,A,e,r){var P,i,D,u,Q;u=(A.b-A.a)/10,Q=(A.d-A.c)/10,P=new n.je(),n.Dm(t.L,n.Xo(t.L,e,r))?(i=-3,D=-3):(i=t.p[e],D=t.p[r]),n.Vd(t,i),P.a=A.a,P.c=A.c,P.b=A.a+u*2,P.d=A.c+Q*2,t.qb(P),P.a=A.a+u*4,P.c=A.c+Q*4,P.b=A.a+u*5,P.d=A.c+Q*5,t.qb(P),n.Vd(t,D),P.a=A.a+u*5,P.c=A.c+Q*5,P.b=A.a+u*6,P.d=A.c+Q*6,t.qb(P),P.a=A.a+u*8,P.c=A.c+Q*8,P.b=A.b,P.d=A.d,t.qb(P),n.Vd(t,t.Q);},n.jd=function(t,A,e){t.ub(A-t.V/2,e-t.V/2,t.V);},n.kd=function(t,A,e,r){n.Dm(t.L,n.Xo(t.L,e,r))?(n.Vd(t,-3),t.qb(A),n.Vd(t,t.Q)):t.p[e]!=t.p[r]?n.gd(t,A,e,r):t.p[e]!=0?(n.Vd(t,t.p[e]),t.qb(A),n.Vd(t,t.Q)):t.qb(A);},n.ld=function(t,A,e,r,P,i,D){D.a=A.a+P,D.c=A.c+i,D.b=A.b+P,D.d=A.d+i,n.kd(t,D,e,r);},n.md=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g;for(y=A.b-A.a,b=A.d-A.c,Q=d.Math.sqrt(y*y+b*b),l=2*Jt(b1(d.Math.round(Q/(4*t.Y)))),T=y/(l-1),g=b/(l-1),n.Dm(t.L,n.Xo(t.L,e,r))?(P=-3,i=-3):(P=t.p[e],i=t.p[r]),L=A.a-t.Y/2,c=A.c-t.Y/2,n.Vd(t,P),u=0;u<(l/2|0);u++)t.ub(L,c,t.Y),L+=T,c+=g;for(n.Vd(t,i),D=0;D<(l/2|0);D++)t.ub(L,c,t.Y),L+=T,c+=g;n.Vd(t,t.Q);},n.nd=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v;Q=(A.c-A.d)/9,c=(A.b-A.a)/9,l=A.b+Q,b=A.d+c,L=A.b-Q,g=A.d-c,y=(A.a+l)/2,k=(A.c+b)/2,T=(A.a+L)/2,v=(A.c+g)/2,D=new ht.iO(3),u=new ht.iO(4),ht.dO(D,A.a,A.c),ht.dO(D,y,k),ht.dO(D,T,v),ht.dO(u,T,v),ht.dO(u,y,k),ht.dO(u,l,b),ht.dO(u,L,g),n.Dm(t.L,n.Xo(t.L,e,r))?(P=-3,i=-3):(P=t.p[e],i=t.p[r],t.L.O!=1&&(i=n.qd(t,e),P==n.Cl(t.L,e)&&(P=i))),n.Vd(t,P),t.sb(D),n.Vd(t,i),t.sb(u),n.Vd(t,t.Q);},n.od=function(t,A){var e,r;for(r=0;r<t.$.a.length;r++)t.v=ht.nO(t.v,U.xj(t.$,r));n.pd(t,A),e=.1*A,t.v.c-=e,t.v.d-=e,t.v.b+=2*e,t.v.a+=2*e;},n.pd=function(t,A){var e,r,P,i,D,u,Q;for(P=B(o.vfb,Tt,6,t.L.q,16,1),r=0;r<t.L.r;r++)n.Bm(t.L,r)&&(P[n.Zl(t.L,0,r)]=true,P[n.Zl(t.L,1,r)]=true);for(D=new ht.oO(),e=0;e<t.L.q;e++)i=n.Jm(t.L,e)?A*.47:P[e]?A*.38:0,i!=0&&(u=n.Ai(t.R,n.Ql(t.L,e)),Q=n.Bi(t.R,n.Rl(t.L,e)),ht.mO(D,u-i,Q-i,i*2,i*2),t.v=ht.nO(t.v,D));},n.qd=function(t,A){var e,r;if((t.G&4224)!=0||(r=n.rd(t,A),r==-1&&(e=n.Fo(t.L,A),e!=-1&&(A=e,r=n.rd(t,A))),r==-1))return t.p[A];switch(r&255){case 1:return 384;case 2:return 64;default:return 448;}},n.rd=function(t,A){var e,r,P;return P=-1,r=-1,(t.G&128)!=0||(n.zm(t.L,A)&&(P=n.Hl(t.L,A),r=n.Gl(t.L,A)),e=n.Ho(t.L,A),e!=-1&&(P=n.cm(t.L,e),r=n.bm(t.L,e)),P!=-1&&P!=0&&(P|=r<<8)),P;},n.sd=function(t){var A,e,r,P;for(t.Ab(2*t.S),P=new n.je(),r=0;r<t.L.r;r++)A=n.Zl(t.L,0,r),e=n.Zl(t.L,1,r),n.Bm(t.L,r)&&(P.a=n.Ai(t.R,n.Ql(t.L,A)),P.c=n.Bi(t.R,n.Rl(t.L,A)),P.b=n.Ai(t.R,n.Ql(t.L,e)),P.d=n.Bi(t.R,n.Rl(t.L,e)),n.Vd(t,-2),t.qb(P));},n.td=function(t){var A,e,r,P,i,D;if(t.L.K){for(D=t.W,n.Vd(t,-7),A=0;A<t.L.f;A++)n.Jm(t.L,A)&&t.ub(n.Ai(t.R,n.Ql(t.L,A))-D,n.Bi(t.R,n.Rl(t.L,A))-D,2*D);for(t.Ab(2*t.W),i=new n.je(),P=0;P<t.L.r;P++)e=n.Zl(t.L,0,P),r=n.Zl(t.L,1,P),pt(J(J(n.Ol(t.L,e),n.Ol(t.L,r)),i0),0)&&(i.a=n.Ai(t.R,n.Ql(t.L,e)),i.c=n.Bi(t.R,n.Rl(t.L,e)),i.b=n.Ai(t.R,n.Ql(t.L,r)),i.d=n.Bi(t.R,n.Rl(t.L,r)),t.qb(i));}},n.ud=function(t){var A,e,r,P;if(t.L.K){if(n.Vd(t,320),(t.G&8)!=0)for(A=0;A<t.L.f;A++)pt(J(n.Ol(t.L,A),-536870913),0)&&t.ub(n.Ai(t.R,n.Ql(t.L,A))-t.Z/2,n.Bi(t.R,n.Rl(t.L,A))-t.Z/2,t.Z);for(P=0;P<t.L.g;P++)n.hm(t.L,P)!=0&&(e=n.Zl(t.L,0,P),r=n.Zl(t.L,1,P),t.ub((n.Ai(t.R,n.Ql(t.L,e))+n.Ai(t.R,n.Ql(t.L,r))-t.Z)/2,(n.Bi(t.R,n.Rl(t.L,e))+n.Bi(t.R,n.Rl(t.L,r))-t.Z)/2,t.Z));}},n.vd=function(t){t.J=1,t.R=new n.Ci(),t.$=new U.Mj(),t.U=new U.Mj(),t.s=B(o.vfb,Tt,6,t.L.q,16,1),t.w=new ht.aO(),t.Q=0,t.C=-1,n._d(t);},n.wd=function(t,A){var e;if(n.Lo(t.L,A)==0)return  false;for(e=0;e<n.Lo(t.L,A);e++)if(!n.Dm(t.L,n._o(t.L,A,e)))return  false;return  true;},n.xd=function(t){var A;for(t.q=B(o.vfb,Tt,6,t.L.q,16,1),A=0;A<t.L.r;A++)t.q[n.Zl(t.L,0,A)]=true,t.q[n.Zl(t.L,1,A)]=true;},n.yd=function(t,A){var e;if(n.$o(t.L,A)!=2)return  false;for(e=0;e<2;e++)if(n.ap(t.L,A,e)!=2)return  false;return  true;},n.zd=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b;for(T=false,P.a=0,P.b=0,r>0?i=$r:i=ar,b=n.Yl(t.L,A,e),L=0;L<n.$o(t.L,A);L++)D=n._o(t.L,A,L),u=b,n.Zl(t.L,0,D)==A?y=n.Zl(t.L,1,D):y=n.Zl(t.L,0,D),y!=e&&(c=n.Yl(t.L,A,y),u<c&&(u+=N1),Q=u-c,r>0?(Q<A1&&(T=true),Q>$r&&(Q=$r),Q<.523598776&&(Q=.523598776),Q<=i&&(i=Q,l=t.T*d.Math.tan(i-v0)/2,P.a=-(l*d.Math.sin(u)),P.b=-(l*d.Math.cos(u)))):(Q>=A1&&(T=true),Q<ar&&(Q=ar),Q>5.759586531&&(Q=5.759586531),Q>=i&&(i=Q,l=t.T*d.Math.tan(4.712388981-i)/2,P.a=-(l*d.Math.sin(u)),P.b=-(l*d.Math.cos(u)))));return T;},n.Ad=function(t,A,e,r){var P;if(A==0){e<0?r.a=t.T:r.a=-t.T,r.b=0;return;}P=d.Math.atan(e/A),A<0&&(P+=A1),r.a=-(t.T*d.Math.sin(P)),r.b=t.T*d.Math.cos(P);},n.Bd=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g;P=new n.je(),Q=new n.je(),L=new ht.aO(),l=new ht.aO(),D=n.Zl(t.L,0,e),u=n.Zl(t.L,1,e),r&&(T=A.a,A.a=A.b,A.b=T,T=A.c,A.c=A.d,A.d=T,c=D,D=u,u=c),n.Rd(t,A)&&(n.gq(t.L,e)?(P.a=A.a,P.c=A.c,P.b=A.b,P.d=A.d,y=r?-n.Dp(t.L,e):n.Dp(t.L,e),y==0&&(y=1),n.Ad(t,A.b-A.a,A.d-A.c,L),y>0?(Q.a=A.a+L.a,Q.c=A.c+L.b,Q.b=A.b+L.a,Q.d=A.d+L.b,(n.zd(t,D,u,1,l)||n.$o(t.L,D)>1)&&(Q.a+=l.a+L.b,Q.c+=l.b-L.a)):(Q.a=A.a-L.a,Q.c=A.c-L.b,Q.b=A.b-L.a,Q.d=A.d-L.b,(n.zd(t,D,u,-1,l)||n.$o(t.L,D)>1)&&(Q.a+=l.a+L.b,Q.c+=l.b-L.a)),n.im(t.L,e)==386&&n.Qd(P,Q),n.Rd(t,P)&&n.kd(t,P,D,u),n.im(t.L,e)==8?n.Rd(t,Q)&&n.hd(t,Q,D,u):n.Rd(t,Q)&&n.kd(t,Q,D,u)):(n.Ad(t,A.b-A.a,A.d-A.c,L),b=L.a/2,g=L.b/2,i=false,P.a=A.a+b,P.c=A.c+g,P.b=A.b+b,P.d=A.d+g,n.$o(t.L,D)>1&&(n.zd(t,D,u,1,l)?(P.a+=l.a,P.c+=l.b,n.$o(t.L,D)==2&&(l.a!=0||l.b!=0)&&(P.a+=L.b,P.c-=L.a)):t.o[D]=new ht.bO(P.a,P.c)),Q.a=A.a-b,Q.c=A.c-g,Q.b=A.b-b,Q.d=A.d-g,n.$o(t.L,D)>1&&(n.zd(t,D,u,0,l)?(Q.a+=l.a,Q.c+=l.b,n.$o(t.L,D)==2&&(l.a!=0||l.b!=0)&&(Q.a+=L.b,Q.c-=L.a)):(t.o[D]=new ht.bO(Q.a,Q.c),i=true)),n.im(t.L,e)==386&&n.Qd(P,Q),n.im(t.L,e)==8?i?(n.hd(t,P,D,u),n.kd(t,Q,D,u)):(n.kd(t,P,D,u),n.hd(t,Q,D,u)):(n.kd(t,P,D,u),n.kd(t,Q,D,u))));},n.Cd=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b;for(L=t.Q,y=t.F,(t.G&N0)!=0&&(t.Q=-6,t.F=Ie,n.Vd(t,1)),t.o=B(o.Xab,R1,55,t.L.q,0,1),Q=0;Q<t.L.r;Q++)(n.im(t.L,Q)==2||n.im(t.L,Q)==386||n.im(t.L,Q)==8)&&n.Fd(t,Q);for(l=0;l<t.L.r;l++)n.im(t.L,l)!=2&&n.im(t.L,l)!=386&&n.im(t.L,l)!=8&&n.Fd(t,l);if((t.G&64)==0)for(u=0;u<t.L.r;u++)n.am(t.L,u)!=0&&(P=null,n.am(t.L,u)==1||n.am(t.L,u)==2?(n.fm(t.L,u)==2||n.cm(t.L,u)==0||A[n.cm(t.L,u)][n.bm(t.L,u)]>1)&&(n.am(t.L,u)==1?P=n.fm(t.L,u)==2?"E":n.Fm(t.L,u)?"p":"P":P=n.fm(t.L,u)==2?"Z":n.Fm(t.L,u)?"m":"M"):P="?",P!=null&&(t.Cb((t.X*2+1)/3|0),n.Vd(t,n.Dm(t.L,u)?-3:t.L.O==1||(t.G&Z1)!=0?t.Q:448),e=n.Zl(t.L,0,u),r=n.Zl(t.L,1,u),c=(n.Ai(t.R,n.Ql(t.L,e))+n.Ai(t.R,n.Ql(t.L,r)))/2,b=(n.Bi(t.R,n.Rl(t.L,e))+n.Bi(t.R,n.Rl(t.L,r)))/2,i=(n.Ai(t.R,n.Ql(t.L,e))-n.Ai(t.R,n.Ql(t.L,r)))/3,D=(n.Bi(t.R,n.Rl(t.L,e))-n.Bi(t.R,n.Rl(t.L,r)))/3,n.Id(t,c+D,b-i,P,true),n.Vd(t,t.Q),t.Cb(t.X)));if((t.G&4)!=0){for(t.Cb((t.X*2+1)/3|0),n.Vd(t,384),u=0;u<t.L.r;u++)e=n.Zl(t.L,0,u),r=n.Zl(t.L,1,u),T=n.Yp(t.L,u)?"d":n.Up(t.L,u)?"a":"",c=(n.Ai(t.R,n.Ql(t.L,e))+n.Ai(t.R,n.Ql(t.L,r)))/2,b=(n.Bi(t.R,n.Rl(t.L,e))+n.Bi(t.R,n.Rl(t.L,r)))/2,n.Id(t,c,b,T+(""+u),true);n.Vd(t,t.Q),t.Cb(t.X);}(t.G&N0)!=0&&(t.Q=L,t.F=y);},n.Dd=function(t){var A,e;for(e=new U.rrb(t.U);e.a<e.c.a.length;)A=U.qrb(e),n.Vd(t,A.a),n.jd(t,A.b,A.c);n.Vd(t,t.Q);},n.Ed=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt;if(t.K||t.yb(A,n.Il(t.L,A),n.Ai(t.R,n.Ql(t.L,A)),n.Bi(t.R,n.Rl(t.L,A))),Ut=null,n.Bl(t.L,A)!=0&&(Kt=d.Math.abs(n.Bl(t.L,A))==1?"":""+d.Math.abs(n.Bl(t.L,A)),Ut=n.Bl(t.L,A)<0?Kt+"-":Kt+"+"),z=null,wt=n.Ol(t.L,A),nt(wt,0)!=0&&(pt(J(wt,Ae),0)&&(z=z==null?"*":z+",*"),pt(J(wt,ne),0)&&(z=z==null?"!*":z+",!*"),pt(J(wt,ee),0)?z=z==null?"ha":z+",ha":pt(J(wt,2),0)?z=z==null?"a":z+",a":pt(J(wt,4),0)&&(z=z==null?"!a":z+",!a"),pt(J(wt,Z1),0)&&(z=z==null?"s":z+",s"),pt(J(wt,X0),0)&&(V=J(wt,X0),nt(V,1792)==0?z=z==null?"h0":z+",h0":nt(V,1664)==0?z=z==null?"h1":z+",h1":nt(V,1408)==0?z=z==null?"h2":z+",h2":nt(V,128)==0?z=z==null?"h>0":z+",h>0":nt(V,384)==0?z=z==null?"h>1":z+",h>1":nt(V,896)==0?z=z==null?"h>2":z+",h>2":nt(V,f0)==0?z=z==null?"h<3":z+",h<3":nt(V,1536)==0?z=z==null?"h<2":z+",h<2":nt(V,1152)==0&&(z=z==null?"h1-2":z+",h1-2")),pt(J(wt,mA),0)&&(Q=J(wt,mA),nt(Q,re)==0?z=z==null?"c0":z+",c0":nt(Q,Qn)==0?z=z==null?"c+":z+",c+":nt(Q,on)==0&&(z=z==null?"c-":z+",c-")),pt(J(wt,$A),0)&&(bt=J(wt,$A),nt(bt,98304)==0?z=z==null?"pi0":z+",pi0":nt(bt,81920)==0?z=z==null?"pi1":z+",pi1":nt(bt,49152)==0?z=z==null?"pi2":z+",pi2":nt(bt,N0)==0&&(z=z==null?"pi>0":z+",pi>0")),pt(J(wt,aA),0)&&(ot=J(wt,aA),nt(ot,Pe)==0?z=z==null?"n1":z+",n1":nt(ot,Mn)==0?z=z==null?"n2":z+",n2":nt(ot,Rn)==0?z=z==null?"n3":z+",n3":nt(ot,ie)==0?z=z==null?"n<3":z+",n<3":nt(ot,j0)==0?z=z==null?"n<4":z+",n<4":nt(ot,tn)==0?z=z==null?"n>1":z+",n>1":nt(ot,iA)==0?z=z==null?"n>2":z+",n>2":nt(ot,ln)==0?z=z==null?"n>3":z+",n>3":nt(ot,3276800)==0?z=z==null?"n1-2":z+",n1-2":nt(ot,2228224)==0?z=z==null?"n1-3":z+",n1-3":nt(ot,2490368)==0&&(z=z==null?"n2-3":z+",n2-3")),pt(J(wt,Ln),0)&&(b=J(wt,Ln),nt(b,Oe)==0?z=z==null?"e0":z+",e0":nt(b,xe)==0?z=z==null?"e1":z+",e1":nt(b,Ne)==0?z=z==null?"e2":z+",e2":nt(b,Ge)==0?z=z==null?"e3":z+",e3":nt(b,tP)==0?z=z==null?"e<2":z+",e<2":nt(b,AP)==0?z=z==null?"e<3":z+",e<3":nt(b,fe)==0?z=z==null?"e<4":z+",e<4":nt(b,Cn)==0?z=z==null?"e>0":z+",e>0":nt(b,ze)==0?z=z==null?"e>1":z+",e>1":nt(b,Je)==0?z=z==null?"e>2":z+",e>2":nt(b,De)==0?z=z==null?"e>3":z+",e>3":nt(b,nP)==0?z=z==null?"e1-2":z+",e1-2":nt(b,eP)==0?z=z==null?"e1-3":z+",e1-3":nt(b,rP)==0&&(z=z==null?"e2-3":z+",e2-3")),pt(J(wt,120),0)&&(ct=J(wt,120),nt(ct,112)==0?z=z==null?"!r":z+",!r":nt(ct,8)==0?z=z==null?"r":z+",r":nt(ct,96)==0?z=z==null?"rb<3":z+",rb<3":nt(ct,104)==0?z=z==null?"rb2":z+",rb2":nt(ct,88)==0?z=z==null?"rb3":z+",rb3":nt(ct,56)==0&&(z=z==null?"rb4":z+",rb4")),pt(J(wt,vA),0)&&(z=z==null?"r"+U0(A0(J(wt,vA),22)):z+","+("r"+U0(A0(J(wt,vA),22)))),pt(J(wt,EA),0)&&(z=n.cd(z,n.fd(wt))),pt(J(wt,NA),0)&&(z=z==null?"f":z+",f")),n.Ml(t.L,A)!=0&&(z=n.cd(z,""+n.Ml(t.L,A))),Ht=0,n.Pl(t.L,A)!=0)switch(n.Pl(t.L,A)){case 16:Ut=Ut==null?"|":Ut+",|";break;case 32:Ht=1;break;case 48:Ht=2;}if(y=null,(t.G&64)==0){if(n.wm(t.L,A))y="?";else if(n.Al(t.L,A)!=0&&(n.Hl(t.L,A)==0||e==null||e[n.Hl(t.L,A)][n.Gl(t.L,A)]>1))if(n.$o(t.L,A)==2)switch(n.Al(t.L,A)){case 2:y=n.ym(t.L,A)?"p":"P";break;case 1:y=n.ym(t.L,A)?"m":"M";break;default:y="*";}else switch(n.Al(t.L,A)){case 1:y=n.ym(t.L,A)?"r":"R";break;case 2:y=n.ym(t.L,A)?"s":"S";break;default:y="*";}}if((t.G&768)!=0&&(y=n.cd(y,""+n.mw(t.L,A))),ut=null,(t.G&16)!=0&&n.Ll(t.L,A)!=0&&(ut=""+n.Ll(t.L,A)),k=null,n.Ip(t.L,A)!=-1&&(g=n.rd(t,A),g!=-1&&(k=g==0?"abs":((g&255)==1?"&":"or")+(1+(g>>8)))),G=0,(t.G&L0)==0&&(t.L.K?pt(J(n.Ol(t.L,A),B1),0)&&(G=n.op(t.L,A)):(n.Tl(t.L,A)!=6||n.Ml(t.L,A)!=0||!t.q[A]||n.Pl(t.L,A)!=0)&&(G=n.op(t.L,A))),Qt=false,tt=false,i=(t.G&H1)!=0?null:n.El(t.L,A),i!=null&&h.akb(w.SAb(i).substr(0,1),"]")&&(Qt=z==null,tt=true,z=n.cd((w.NAb(1,w.SAb(i).length+1),w.SAb(i).substr(1)),z),i=null),i!=null?G=0:n.Jl(t.L,A)!=null?(P=pt(J(n.Ol(t.L,A),1),0)?"[!":"[",i=P+n.Kl(t.L,A)+"]",w.SAb(i).length>5&&(i=P+n.Jl(t.L,A).length+"]"),pt(J(n.Ol(t.L,A),B1),0)&&(G=-1)):pt(J(n.Ol(t.L,A),1),0)?(i="?",pt(J(n.Ol(t.L,A),B1),0)&&(G=-1)):(n.Tl(t.L,A)!=6||Ut!=null||z!=null&&(!Qt||(t.G&LA)==0)||G>0||!t.q[A])&&(i=n.Il(t.L,A)),lt=0,!n.Pm(t.L,A)&n.Jm(t.L,A)&&n.Vd(t,-8),i!=null?(lt=t.vb(i),n.Id(t,n.Ai(t.R,n.Ql(t.L,A)),n.Bi(t.R,n.Rl(t.L,A)),i,true),t.s[A]=true):n.yd(t,A)&&n.Hd(t,n.Ai(t.R,n.Ql(t.L,A)),n.Bi(t.R,n.Rl(t.L,A)),A),Ut!=null&&(t.Cb((t.X*2+1)/3|0),xt=n.Ai(t.R,n.Ql(t.L,A))+((lt+t.vb(Ut))/2+1),kt=n.Bi(t.R,n.Rl(t.L,A))-((t.wb()*4-4)/8|0),n.Id(t,xt,kt,Ut,true),t.Cb(t.X)),(t.G&2)!=0&&(z=""+A),z!=null&&(tt?t.Cb((t.X*5+1)/6|0):t.Cb((t.X*2+1)/3|0),xt=n.Ai(t.R,n.Ql(t.L,A))-(lt+t.vb(z))/2,kt=n.Bi(t.R,n.Rl(t.L,A))-((t.wb()*4-4)/8|0),n.Id(t,xt,kt,z,true),t.Cb(t.X)),y!=null&&(t.Cb((t.X*2+1)/3|0),xt=n.Ai(t.R,n.Ql(t.L,A))-(lt+t.vb(y))/2,kt=n.Bi(t.R,n.Rl(t.L,A))+((t.wb()*4+4)/8|0),Et=t.C,t.L.O!=1&&(t.G&Z1)==0&&n.Vd(t,448),n.Id(t,xt,kt,y,false),n.Vd(t,Et),t.Cb(t.X)),ut!=null&&(t.Cb((t.X*2+1)/3|0),xt=n.Ai(t.R,n.Ql(t.L,A))+((lt+t.vb(ut))/2+1),kt=n.Bi(t.R,n.Rl(t.L,A))+((t.wb()*4+4)/8|0),Et=t.C,n.Vd(t,n.Am(t.L,A)?384:448),n.Id(t,xt,kt,ut,true),n.Vd(t,Et),t.Cb(t.X)),k!=null&&(r=n.Md(t,A),t.Cb((t.X*2+1)/3|0),xt=n.Ai(t.R,n.Ql(t.L,A))+.7*t.wb()*d.Math.sin(r),kt=n.Bi(t.R,n.Rl(t.L,A))+.7*t.wb()*d.Math.cos(r),Et=t.C,!t.K&&t.L.O!=1&&n.Vd(t,n.qd(t,A)),n.Id(t,xt,kt,k,false),n.Vd(t,Et),t.Cb(t.X)),G==0&&Ht==0){t.C==-8&&n.Vd(t,-9);return;}for(E=B(o.O7,Ft,6,4,15,1),O=0;O<n.Mo(t.L,A);O++)for(u=n._o(t.L,A,O),it=0;it<2;it++)n.Zl(t.L,it,u)==A&&(It=n.Yl(t.L,n.Zl(t.L,it,u),n.Zl(t.L,1-it,u)),It<ue?(E[0]-=It+v0,E[3]+=It+A1):It<0?(E[2]+=It+v0,E[3]-=It):It<v0?(E[1]+=It,E[2]+=v0-It):(E[0]+=It-v0,E[1]+=A1-It));if(n.$o(t.L,A)==0?n.Hm(t.L,A)?E[3]-=.2:E[1]-=.2:E[1]-=.1,(Ut!=null||ut!=null)&&(E[1]+=10),(z!=null||y!=null)&&(E[3]+=10),S="",G!=0&&(q=t.vb("H"),p=0,v=t.wb(),G==-1?(S="n",t.Cb((t.X*2+1)/3|0),p=t.vb(S)):G>1&&(S=""+G,t.Cb((t.X*2+1)/3|0),p=t.vb(S)),E[1]<.6||E[3]<.6?(L=n.Bi(t.R,n.Rl(t.L,A)),E[1]<=E[3]?(E[1]+=10,l=n.Ai(t.R,n.Ql(t.L,A))+(lt+q)/2):(E[3]+=10,l=n.Ai(t.R,n.Ql(t.L,A))-(lt+q)/2-p)):(l=n.Ai(t.R,n.Ql(t.L,A)),E[0]<E[2]?(E[0]+=10,L=n.Bi(t.R,n.Rl(t.L,A))-v):(E[2]+=10,L=n.Bi(t.R,n.Rl(t.L,A))+v)),p>0&&(xt=l+(q+p)/2,kt=L+((t.wb()*4+4)/8|0),n.Id(t,xt,kt,S,true),t.Cb(t.X)),n.Id(t,l,L,"H",true)),D=0,Ht!=0){for(Dt=50,T=0,K=0;K<4;K++)c=K>1?K-2:K+2,E[K]<Dt?(D=K,Dt=E[K],T=E[c]):E[K]==Dt&&E[c]>T&&(D=K,T=E[c]);switch(D){case 0:l=n.Ai(t.R,n.Ql(t.L,A)),L=n.Bi(t.R,n.Rl(t.L,A))-t.V-lt/2;break;case 1:l=n.Ai(t.R,n.Ql(t.L,A))+t.V+lt/2,L=n.Bi(t.R,n.Rl(t.L,A));break;case 2:l=n.Ai(t.R,n.Ql(t.L,A)),L=n.Bi(t.R,n.Rl(t.L,A))+t.V+lt/2;break;default:l=n.Ai(t.R,n.Ql(t.L,A))-t.V-lt/2,L=n.Bi(t.R,n.Rl(t.L,A));}if(Ht==1)U.sj(t.$,new ht.pO(l-t.V,L-t.V,2*t.V,2*t.V)),t.K||U.sj(t.U,new n.ie(l,L,n.wd(t,A)?-3:t.p[A]));else {switch(D){case 2:case 0:Nt=2*t.V,Bt=0,l-=t.V;break;case 1:Nt=0,Bt=2*t.V,L-=t.V;break;default:Nt=0,Bt=2*t.V,L-=t.V;}U.sj(t.$,new ht.pO(l-t.V,L-t.V,2*t.V,2*t.V)),t.K||U.sj(t.U,new n.ie(l,L,n.wd(t,A)?-3:t.p[A])),U.sj(t.$,new ht.pO(l+Nt-t.V,L+Bt-t.V,2*t.V,2*t.V)),t.K||U.sj(t.U,new n.ie(l+Nt,L+Bt,n.wd(t,A)?-3:t.p[A]));}}t.C==-8&&n.Vd(t,-9);},n.Fd=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;if(p=new n.je(),e=new n.je(),D=new n.je(),k=new ht.aO(),g=new ht.aO(),P=n.Zl(t.L,0,A),i=n.Zl(t.L,1,A),t.zb(A,n.Ai(t.R,n.Ql(t.L,P)),n.Bi(t.R,n.Rl(t.L,P)),n.Ai(t.R,n.Ql(t.L,i)),n.Bi(t.R,n.Rl(t.L,i))),!n.Pm(t.L,P)&&!n.Pm(t.L,i)&&pt(J(Pt(n.Ol(t.L,P),n.Ol(t.L,i)),i0),0)&&n.Vd(t,-8),t.o[P]?(p.a=t.o[P].a,p.c=t.o[P].b):(p.a=n.Ai(t.R,n.Ql(t.L,P)),p.c=n.Bi(t.R,n.Rl(t.L,P))),t.o[i]?(p.b=t.o[i].a,p.d=t.o[i].b):(p.b=n.Ai(t.R,n.Ql(t.L,i)),p.d=n.Bi(t.R,n.Rl(t.L,i))),(n.hm(t.L,A)&sA)!=0){n.Rd(t,p)&&t.rb(p),n.Vd(t,-9);return;}switch(u=n.im(t.L,A)==8?0:n.im(t.L,A)==16?1:n.fm(t.L,A),u){case 1:if(Q=n.im(t.L,A),(t.G&128)!=0&&(Q==257||Q==129)&&(S=n.Zl(t.L,0,A),c=n.Hl(t.L,S),c!=0)){for(T=n.Gl(t.L,S),y=0,r=0;r<t.L.f;r++)n.Hl(t.L,r)==c&&n.Gl(t.L,r)==T&&++y;y==1&&(Q=1);}switch(Q){case 1:n.Rd(t,p)&&n.kd(t,p,P,i);break;case 257:n.Od(t,p,P,i);break;case 129:for(E=p.b-p.a,q=p.d-p.c,n.Dm(t.L,n.Xo(t.L,P,i))?(l=-3,L=-3):(l=t.p[P],L=n.qd(t,P),l==n.Cl(t.L,P)&&(l=L)),b=2;b<17;b+=2)e.a=p.a+b*E/17-b*q/128,e.c=p.c+b*q/17+b*E/128,e.b=p.a+b*E/17+b*q/128,e.d=p.c+b*q/17-b*E/128,n.Rd(t,e)&&(n.Vd(t,b<9?l:L),t.qb(e),n.Vd(t,t.Q));break;case 16:n.Rd(t,p)&&n.md(t,p,P,i);}break;case 0:case 2:if((t.s[P]||n.Qo(t.L,P)==2)&&(t.s[i]||n.Qo(t.L,i)==2)&&!n.gq(t.L,A)&&u==2){if(!n.Rd(t,p))break;n.Ad(t,p.b-p.a,p.d-p.c,k),E=k.a/2,q=k.b/2,e.a=p.a+E,e.c=p.c+q,e.b=p.b+E,e.d=p.d+q,D.a=p.a-E,D.c=p.c-q,D.b=p.b-E,D.d=p.d-q,n.im(t.L,A)==386&&n.Qd(e,D),n.kd(t,e,P,i),u==2?n.kd(t,D,P,i):n.hd(t,D,P,i);}else (t.s[i]||n.Qo(t.L,i)==2)&&u==2?n.Bd(t,p,A,false):(t.s[P]||n.Qo(t.L,P)==2)&&u==2?n.Bd(t,p,A,true):(v=n.Dp(t.L,A),v==0&&(v=1),e.a=p.a,e.c=p.c,e.b=p.b,e.d=p.d,n.Ad(t,p.b-p.a,p.d-p.c,k),v>0?(D.a=p.a+k.a,D.c=p.c+k.b,D.b=p.b+k.a,D.d=p.d+k.b,(n.zd(t,P,i,1,g)||n.$o(t.L,P)>1)&&(D.a+=g.a+k.b,D.c+=g.b-k.a),(n.zd(t,i,P,-1,g)||n.$o(t.L,i)>1)&&(D.b+=g.a-k.b,D.d+=g.b+k.a)):(D.a=p.a-k.a,D.c=p.c-k.b,D.b=p.b-k.a,D.d=p.d-k.b,(n.zd(t,P,i,-1,g)||n.$o(t.L,P)>1)&&(D.a+=g.a+k.b,D.c+=g.b-k.a),(n.zd(t,i,P,1,g)||n.$o(t.L,i)>1)&&(D.b+=g.a-k.b,D.d+=g.b+k.a)),n.im(t.L,A)==386&&n.Qd(e,D),n.Rd(t,e)&&n.kd(t,e,P,i),u==2?n.Rd(t,D)&&n.kd(t,D,P,i):n.Rd(t,D)&&n.hd(t,D,P,i));break;case 3:n.Rd(t,p)&&(n.kd(t,p,P,i),n.Ad(t,p.b-p.a,p.d-p.c,k),n.ld(t,p,P,i,k.a,k.b,e),n.ld(t,p,P,i,-k.a,-k.b,e));break;case 4:n.Rd(t,p)&&(n.Ad(t,p.b-p.a,p.d-p.c,k),n.ld(t,p,P,i,1.5*k.a,1.5*k.b,e),n.ld(t,p,P,i,.5*k.a,.5*k.b,e),n.ld(t,p,P,i,-0.5*k.a,-0.5*k.b,e),n.ld(t,p,P,i,-1.5*k.a,-1.5*k.b,e));break;case 5:n.Rd(t,p)&&(n.kd(t,p,P,i),n.Ad(t,p.b-p.a,p.d-p.c,k),n.ld(t,p,P,i,2*k.a,2*k.b,e),n.ld(t,p,P,i,k.a,k.b,e),n.ld(t,p,P,i,-k.a,-k.b,e),n.ld(t,p,P,i,-2*k.a,-2*k.b,e));}t.C==-8&&n.Vd(t,-9);},n.Gd=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g;for(c=false,r=0;r<t.L.g;r++)l=null,n.Cm(t.L,r)?(y=n._l(t.L,r),L=n.$l(t.L,r),l=y==L?"["+y+"]":"["+y+":"+L+"]"):(n.hm(t.L,r)&Qe)!=0?l=(n.hm(t.L,r)&Qe)==j0?"a":(n.hm(t.L,r)&384)==256?"r!a":"!a":(n.hm(t.L,r)&384)!=0&&(l=(n.hm(t.L,r)&384)==256?"r":"!r"),T=(n.hm(t.L,r)&iA)>>17,T!=0&&(l=(l??"")+T),l!=null&&(A=n.Zl(t.L,0,r),e=n.Zl(t.L,1,r),c||(t.Cb((t.X*2+1)/3|0),c=true),b=(n.Ai(t.R,n.Ql(t.L,A))+n.Ai(t.R,n.Ql(t.L,e)))/2,g=(n.Bi(t.R,n.Rl(t.L,A))+n.Bi(t.R,n.Rl(t.L,e)))/2,i=n.Ai(t.R,n.Ql(t.L,e))-n.Ai(t.R,n.Ql(t.L,A)),D=n.Bi(t.R,n.Rl(t.L,e))-n.Bi(t.R,n.Rl(t.L,A)),P=d.Math.sqrt(i*i+D*D),Q=.6*t.vb(l),u=.55*t.wb(),P!=0&&(i>0?n.Id(t,b+Q*D/P,g-u*i/P,l,true):n.Id(t,b-Q*D/P,g+u*i/P,l,true)));c&&t.Cb(t.X);},n.Hd=function(t,A,e,r){U.sj(t.$,new ht.pO(A-t.V,e-t.V,2*t.V,2*t.V)),t.K||U.sj(t.U,new n.ie(A,e,n.wd(t,r)?-3:t.p[r]));},n.Id=function(t,A,e,r,P){var i,D,u;P&&(i=t.vb(r),D=i/2+(t.wb()/8|0),u=t.wb()/2|0,(r=="+"||r=="-")&&(u=u*2/3),U.sj(t.$,new ht.pO(A-D,e-u,2*D,2*u))),t.K||t.tb(r,A,e);},n.Jd=function(t){var A;A=t.a,t.a=t.b,t.b=A,A=t.c,t.c=t.d,t.d=A;},n.Kd=function(t,A,e){var r;return r=A==0?N1+t[0]-t[t.length-1]:t[A]-t[A-1],e>PP&&e<c0?r-=2*d.Math.cos(e+Sn):r-=.5*d.Math.cos(e+Sn),r;},n.Ld=function(t){var A;return A=new ht.oO(),t.a<=t.b?(A.c=t.a,A.b=t.b-t.a):(A.c=t.b,A.b=t.a-t.b),t.c<=t.d?(A.d=t.c,A.a=t.d-t.c):(A.d=t.d,A.a=t.c-t.d),A;},n.Md=function(t,A){var e,r,P,i,D,u,Q;for(e=B(o.O7,Ft,6,n.Lo(t.L,A),15,1),P=0;P<n.Lo(t.L,A);P++)e[P]=n.Yl(t.L,A,n.Zo(t.L,A,P));for(w.lAb(e,w.SAb(T0(U.Xrb.prototype.ld,U.Xrb,[]))),i=n.Nd(e,0),D=n.Kd(e,0,i),r=1;r<e.length;r++)u=n.Nd(e,r),Q=n.Kd(e,r,u),D<Q&&(D=Q,i=u);return i;},n.Nd=function(t,A){var e;return A>0?(t[A]+t[A-1])/2:(e=A1+(t[0]+t[t.length-1])/2,e>A1?e-N1:e);},n.Od=function(t,A,e,r){var P,i,D,u;if(u=new n.je(),!(A.a==A.b&&A.c==A.d)){for(u.a=A.a,u.c=A.c,u.b=A.b,u.d=A.d,D=n.Ld(u),P=0;P<t.$.a.length;P++)if(i=U.xj(t.$,P),!(i.c>D.c+D.b||i.d>D.d+D.a||D.c>i.c+i.b||D.d>i.d+i.a)){if(n.Pd(t,u.a,u.c,P)){if(n.Pd(t,u.b,u.d,P))return;n.Sd(t,u,0,P),n.Od(t,u,e,r);return;}if(n.Pd(t,u.b,u.d,P)){n.Sd(t,u,1,P),n.Od(t,u,e,r);return;}}n.nd(t,u,e,r);}},n.Pd=function(t,A,e,r){var P;return (t.G&1)!=0?false:(P=U.xj(t.$,r),A>P.c&&A<P.c+P.b&&e>P.d&&e<P.d+P.a);},n.Qd=function(t,A){var e;e=t.b,t.b=A.b,A.b=e,e=t.d,t.d=A.d,A.d=e;},n.Rd=function(t,A){var e,r,P,i,D,u;if(A.a==A.b&&A.c==A.d){for(D=new U.rrb(t.$);D.a<D.c.a.length;)if(i=U.qrb(D),ht.kO(i,A.a,A.c))return  false;return  true;}for(u=n.Ld(A),e=false,A.a>A.b&&(n.Jd(A),e=true),r=0;r<t.$.a.length;r++)if(i=U.xj(t.$,r),!(i.c>u.c+u.b||i.d>u.d+u.a||u.c>i.c+i.b||u.d>i.d+i.a)){if(n.Pd(t,A.a,A.c,r))return n.Pd(t,A.b,A.d,r)?(e&&n.Jd(A),false):(n.Sd(t,A,0,r),P=n.Rd(t,A),e&&n.Jd(A),P);if(n.Pd(t,A.b,A.d,r))return n.Sd(t,A,1,r),P=n.Rd(t,A),e&&n.Jd(A),P;}return e&&n.Jd(A),true;},n.Sd=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b;e==0?(y=A.a,c=A.c,T=A.b,b=A.d):(y=A.b,c=A.d,T=A.a,b=A.c),L=U.xj(t.$,r),Q=T>y?L.c+L.b:L.c,l=b>c?L.d+L.a:L.d,P=T-y,i=b-c,d.Math.abs(P)>d.Math.abs(i)?c==b?(D=Q,u=c):(D=y+P*(l-c)/i,T>y==Q>D?u=l:(D=Q,u=c+i*(Q-y)/P)):y==T?(D=y,u=l):(u=c+i*(Q-y)/P,b>c==l>u?D=Q:(D=y+P*(l-c)/i,u=l)),e==0?(A.a=D,A.c=u):(A.b=D,A.d=u);},n.Td=function(t,A){var e,r,P,i,D,u;if(t.L.q!=0){for(n.bw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),t.B=A,n.ed(t),P=n.gw(t.L),i=false,t.p=B(o.Q7,H,6,t.L.q,15,1),e=0;e<t.L.q;e++)t.p[e]=n.Cl(t.L,e),t.p[e]!=0&&(i=true),n.Pm(t.L,e)&&(t.p[e]=128),n.rm(t.L,e)&&(t.G&B1)==0&&(t.p[e]=256);for(n.Vd(t,-10),n.td(t),n.sd(t),n.ud(t),n.bd(t),t.Cb(t.X),t.Ab(t.Y),n.Vd(t,t.Q),n.xd(t),w.kAb(t.U.a,0),w.kAb(t.$.a,0),(t.G&1)!=0&&(n.Cd(t,P),n.Dd(t),n.Gd(t)),D=0;D<t.L.q;D++)n.wd(t,D)?(n.Vd(t,-3),n.Ed(t,D,P),n.Vd(t,t.Q)):t.p[D]!=0?(n.Vd(t,t.p[D]),n.Ed(t,D,P),n.Vd(t,t.Q)):!i&&t.L.O!=1&&n.Tl(t.L,D)!=1&&n.Tl(t.L,D)!=6&&(t.G&f0)==0&&n.Jl(t.L,D)==null&&n.Tl(t.L,D)<n.Xc.length&&n.Xc[n.Tl(t.L,D)]!=0?(u=n.Xc[n.Tl(t.L,D)],n.Yd(t,(r=t.M!=0?t.M:t.D!=0?t.D:-1,Lt.y4(u,r))),n.Ed(t,D,P),n.Vd(t,t.Q)):n.Ed(t,D,P);(t.G&1)==0&&(n.Dd(t),n.Gd(t),n.Cd(t,P));}},n.Ud=function(t,A,e,r){var P;switch(P=e/2,r&l1){case l1:if(A){t.w.a=A.c+A.b/2,t.w.b=A.d+A.a-P;break;}case 0:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d+t.v.a+P,A&&t.w.b>A.d+A.a-P&&(t.w.b=A.d+A.a-P);break;case UA:if(A){t.w.a=A.c+A.b/2,t.w.b=A.d+P;break;}case yA:t.w.a=t.v.c+t.v.b/2,t.w.b=t.v.d-P,A&&t.w.b<A.d+P&&(t.w.b=A.d+P);}},n.Vd=function(t,A){if(!t.K&&(A==-10&&(t.C=-999,A=t.Q),A!=-2&&A!=-7&&t.N!=0&&(A=-4),A!=t.C&&!(t.C==-8&&A!=-9)))switch(A==-8&&(t.O=t.C),A==-9&&(A=t.O),t.C=A,A){case 0:t.Bb(t.F==0?G0:t.F);break;case  -6:t.Bb(t.F);break;case  -4:t.Bb(t.N);break;case  -2:t.Bb(t.t);break;case  -3:t.Bb(t.u);break;case  -7:t.Bb(t.H);break;case  -8:t.Bb(t.I);break;case  -5:t.Bb(t.P);break;case 64:t.Bb(-14655233);break;case 128:t.Bb(fA);break;case 256:t.Bb(-4194049);break;case 192:t.Bb(-16711936);break;case 320:t.Bb(-24576);break;case 384:t.Bb(-16744448);break;case 448:t.Bb(-6291456);break;case 1:t.Bb(Ie);break;default:t.Bb(G0);}},n.Wd=function(t,A){t.G=A;},n.Xd=function(t,A,e){t.Q=-6,t.F=A,t.D=e,n._d(t);},n.Yd=function(t,A){if(t.N!=0){t.C!=-4&&(t.C=-4,t.Bb(t.N));return;}t.C=-5,t.P=A,t.Bb(A);},n.Zd=function(t){var A,e,r,P,i;for(P=n.Ai(t.R,n.Ql(t.L,0)),e=n.Ai(t.R,n.Ql(t.L,0)),i=n.Bi(t.R,n.Rl(t.L,0)),r=n.Bi(t.R,n.Rl(t.L,0)),A=0;A<t.L.q;A++)P>n.Ai(t.R,n.Ql(t.L,A))&&(P=n.Ai(t.R,n.Ql(t.L,A))),e<n.Ai(t.R,n.Ql(t.L,A))&&(e=n.Ai(t.R,n.Ql(t.L,A))),i>n.Bi(t.R,n.Rl(t.L,A))&&(i=n.Bi(t.R,n.Rl(t.L,A))),r<n.Bi(t.R,n.Rl(t.L,A))&&(r=n.Bi(t.R,n.Rl(t.L,A)));return new ht.pO(P,i,e-P,r-i);},n.$d=function(t,A,e){var r,P;return t.L.q==0?null:(t.v=n.Zd(t),r=t.R.c*n.Ul(t.L),P=new n.Ei(t.v,A,r,e),P.c==1&&P.a==0&&P.b==0?P=null:(n.ui(P,t.R),n.xi(P,t.v)),n.Ud(t,A,r,e),P);},n._d=function(t){var A;A=t.M!=0?t.M:t.D!=0?t.D:-1,t.t=Lt.z4(A),t.u=Lt.y4(Ke,A),t.H=-24321,t.I=-6291392;},n.ae=function(t,A,e,r){var P;return n.be(t,A,e,r),n.zi(t.R)?null:(P=t.R,n.vi(t.R,t.L),t.R=new n.Ci(),P);},n.be=function(t,A,e,r){var P,i,D,u;if(t.L.q==0)return null;for(D=t.Db(e,r),n.bw(t.L,(t.G&256)!=0?63:(t.G&512)!=0?95:31),n.xd(t),w.kAb(t.U.a,0),w.kAb(t.$.a,0),t.B=A,n.ed(t),t.Cb(t.X),t.K=true,i=0;i<t.L.q;i++)n.Ed(t,i,null);return t.K=false,P=t.R.c*n.Ul(t.L),n.od(t,P),n.Ud(t,e,P,r),!e||ht.lO(e,t.v)?D:(u=new n.Ei(t.v,e,P,r),n.ui(u,t.R),n.xi(u,t.v),n.wi(u,t.w),D?(n.ui(u,D),D):u);},n.ce=function(t){n.de.call(this,t,0);},n.de=function(t,A){n.ad(this),this.L=t,this.G=A,this.xb();},x(121,1,{121:1}),s.xb=function(){n.vd(this);},s.yb=function(t,A,e,r){},s.zb=function(t,A,e,r,P){},s.Db=function(t,A){return n.$d(this,t,A);},s.r=0,s.t=0,s.u=0,s.A=0,s.C=0,s.D=0,s.F=0,s.G=0,s.H=0,s.I=0,s.J=0,s.K=false,s.M=0,s.N=0,s.O=0,s.P=0,s.Q=0,s.S=0,s.T=0,s.V=0,s.W=0,s.X=0,s.Y=0,s.Z=0,n.Yc=.15,n.Zc=.06,n.$c=.6,o.V7=N(121),n.ie=function(t,A,e){this.b=t,this.c=A,this.a=e;},x(144,1,{144:1},n.ie),s.a=0,s.b=0,s.c=0,o.T7=N(144),n.je=function(){},x(67,1,{},n.je),s.a=0,s.b=0,s.c=0,s.d=0,o.U7=N(67),n.ke=function(t,A){var e;if(t.i==null)return  false;for(e=0;e<t.i.length;e++)if(!A.Hb(t.i[e].a,t.i[e].b))return  false;return  true;},n.le=function(t,A,e){var r;if(t.i!=null)for(r=0;r<t.i.length;r++)t.i[r].a+=A,t.i[r].b+=e;},n.me=function(t,A){var e;if(t.i!=null)for(e=0;e<t.i.length;e++)t.i[e].a*=A,t.i[e].b*=A;},n.ne=function(t,A){t.g=A;},n.oe=function(t,A,e){var r;if(t.i!=null)for(r=0;r<t.i.length;r++)t.i[r].a=t.o[r]+A-t.k,t.i[r].b=t.p[r]+e-t.n;},n.pe=function(t,A,e){var r,P;if(t.k=A,t.n=e,t.i!=null)for(P=t.i.length,t.o=B(o.O7,Ft,6,P,15,1),t.p=B(o.O7,Ft,6,P,15,1),r=0;r<P;r++)t.o[r]=t.i[r].a,t.p[r]=t.i[r].b;},n.qe=function(t,A,e){var r,P,i;if(t.i!=null)for(r=0;r<t.i.length;r++)i=t.p[r]*A,P=t.o[r]-e,t.i[r].a=t.k+i*d.Math.sin(P),t.i[r].b=t.n+i*d.Math.cos(P);},n.re=function(t,A,e){var r,P,i,D;if(t.k=A,t.n=e,t.i!=null)for(D=t.i.length,t.o=B(o.O7,Ft,6,D,15,1),t.p=B(o.O7,Ft,6,D,15,1),i=0;i<D;i++)r=A-t.i[i].a,P=e-t.i[i].b,t.p[i]=d.Math.sqrt(r*r+P*P),t.o[i]=n.no(A,e,t.i[i].a,t.i[i].b);},x(193,1,{193:1}),s.Jb=function(t){var A,e;if(this.i==null)return null;for(A=new ht.pO(this.i[0].a,this.i[0].b,0,0),e=1;e<this.i.length;e++)A.c>this.i[e].a?(A.b+=A.c-this.i[e].a,A.c=this.i[e].a):A.b<this.i[e].a-A.c&&(A.b=this.i[e].a-A.c),A.d>this.i[e].b?(A.a+=A.d-this.i[e].b,A.d=this.i[e].b):A.a<this.i[e].b-A.d&&(A.a=this.i[e].b-A.d);return A;},s.Nb=function(t){n.me(this,t);},s.pb=function(){var t;return t=new h.Ckb(),h.Bkb(t,iP+this.Lb()+'"'+this.Kb()+Ze),t.a;},s.Ob=function(t,A){n.oe(this,t,A);},s.Pb=function(t,A){n.qe(this,t,A);},s.Qb=function(t,A){n.re(this,t,A);},s.g=false,s.j=false,s.k=0,s.n=0,o.W7=N(193),n.ye=function(t,A,e){if(t.f[A])return 0;if(n.Tl(t.j,A)==7){if(n.Lo(t.j,A)==3)return 6;if(n.$o(t.j,A)==2)return t.a?0:4;}else {if(n.Tl(t.j,A)==8)return 10;if(n.Tl(t.j,A)==15||n.Tl(t.j,A)==33){if(n.$o(t.j,A)==3)return 8;}else if(n.Tl(t.j,A)==16||n.Tl(t.j,A)==34||n.Tl(t.j,A)==52){if(n.$o(t.j,A)==2)return 11;if(n.$o(t.j,A)==3)return n.Bl(t.j,A)==1?12:(e&&n.jn(t.j,A,1),5);}else if(n.Tl(t.j,A)==6)return n.Bl(t.j,A)==-1?n.Lo(t.j,A)==3?16:n.Mo(t.j,A)==3?15:14:(e&&n.jn(t.j,A,-1),n.Lo(t.j,A)!=n.Mo(t.j,A)?2:3);}return 0;},n.ze=function(t,A,e){if(t.f[A])return 0;if(t.a){if(n.Lo(t.j,A)!=3)return 0;}else if(n.Lo(t.j,A)>3)return 0;return n.Tl(t.j,A)==6?(e&&n.jn(t.j,A,1),2):n.Tl(t.j,A)==5?4:0;},n.Ae=function(t,A,e){if(n.Bl(t.j,A)!=0)return 0;if(t.a){if(n.Tl(t.j,A)==5)return n.vp(t.j,A)!=2?0:(e&&n.jn(t.j,A,1),1);if(n.Tl(t.j,A)==7)return n.vp(t.j,A)!=2?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?6:3);if(n.Tl(t.j,A)==8)return n.vp(t.j,A)!=1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?7:4);if(n.Tl(t.j,A)==16)return n.vp(t.j,A)!=1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?5:2);if(n.Tl(t.j,A)==34)return n.vp(t.j,A)!=1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?4:1);}else {if(n.Tl(t.j,A)==5)return n.vp(t.j,A)>2?0:(e&&n.jn(t.j,A,1),1);if(n.Tl(t.j,A)==7)return n.vp(t.j,A)>2?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?5:3);if(n.Tl(t.j,A)==8)return n.vp(t.j,A)>1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?7:4);if(n.Tl(t.j,A)==16)return n.vp(t.j,A)>1?0:(e&&n.jn(t.j,A,-1),n.De(t,A)?5:2);}return 0;},n.Be=function(t,A,e){var r,P,i;if(r=n.Tl(t.j,A),r>=5&&r<=8||r==15||r==16||r==33||r==34||r==52){if(i=n.pp(t.j,A),i!=0)return  true;if(P=n.Bl(t.j,A),r==5&&P>=0)return e&&n.jn(t.j,A,P-1),true;if(r!=5&&P<=0)return e&&n.jn(t.j,A,P+1),true;}return  false;},n.Ce=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b;for(A=0;A<t.j.f;A++)if(t.d[A])for(D=false,u=B(o.Q7,H,6,t.j.f,15,1),b=B(o.Q7,H,6,t.j.f,15,1),Q=B(o.Q7,H,6,t.j.f,15,1),u[0]=A,b[A]=-1,Q[A]=1,P=0,l=0;P<=l&&!D;){for(i=u[P],L=0;L<n.$o(t.j,i)&&!D;L++)if(y=(Q[i]&1)==1^n.fm(t.j,n._o(t.j,i,L))>1,r=n.Zo(t.j,i,L),Q[r]==0&&y)if(t.d[r]){if((Q[i]&1)==1){for(t.d[A]=false,t.d[r]=false,t.k+=2,c=i,T=0;T<Q[i];T++)e=n.Xo(t.j,r,c),t.e[e]&&(t.e[e]=false,--t.c),n.In(t.j,e,n.fm(t.j,e)==1?2:n.fm(t.j,e)-1),r=c,c=b[r];D=true;}}else u[++l]=r,b[r]=i,Q[r]=Q[i]+1;++P;}},n.De=function(t,A){var e;for(e=0;e<n.$o(t.j,A);e++)if(n.Lm(t.j,n.Zo(t.j,A,e)))return  true;return  false;},n.Ee=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;if(A!=null)t.e=A;else for(t.e=B(o.vfb,Tt,6,t.j.g,16,1),T=0;T<t.j.g;T++)n.im(t.j,T)==8&&(t.e[T]=true,n.Mn(t.j,T,1));for(c=0;c<t.j.g;c++)t.e[c]&&++t.c;for(t.d=B(o.vfb,Tt,6,t.j.f,16,1),t.f=B(o.vfb,Tt,6,t.j.f,16,1),u=0;u<t.j.f;u++){for(b=0,g=0;g<n.$o(t.j,u);g++)t.e[n._o(t.j,u,g)]&&++b;b>0&&(t.d[u]=true,++t.b,b==3&&(t.f[u]=true));}for(t.i=B(o.vfb,Tt,6,e.j.a.length,16,1),v=0;v<e.j.a.length;v++)for(t.i[v]=true,l=U.xj(e.k,v),L=0,y=l.length;L<y;++L)if(Q=l[L],!t.e[Q]){t.i[v]=false;break;}for(t.g=B(o.vfb,Tt,6,t.j.f,16,1),k=0;k<e.j.a.length;k++)if(t.i[k]&&U.xj(e.k,k).length!=6)for(P=U.xj(e.j,k),i=0,D=P.length;i<D;++i)r=P[i],U.xj(e.k,k).length==5&&(t.g[r]=true);t.k=0;},n.Fe=function(t,A,e){var r,P;for(P=0;P<n.$o(t.j,A);P++)if(r=n.Zo(t.j,A,P),n.Tl(t.j,r)==8&&n.$o(t.j,r)==1)return e&&n.Bl(t.j,r)==0&&n.jn(t.j,r,-1),true;return  false;},n.Ge=function(t,A,e){var r,P,i,D;if(n.bw(t.j,1),D=new n.Dt(t.j,1),n.Ee(t,A,D),t.c==0)return  true;if(e)for(P=0;P<t.j.f;P++)t.d[P]&&n.Tl(t.j,P)==7&&(t.f[P]||n.$o(t.j,P)==3&&!t.g[P])&&n.jn(t.j,P,1);for(t.a=false,n.Re(t,e),t.j.K&&n.Je(t),n.Te(t,D),n.Oe(t,D),n.Qe(t),n.Ke(t);n.Me(t,D,e);)n.Ke(t);for(;n.Ne(t,D,e);)n.Ke(t);for(;n.Le(t,D,e);)n.Ke(t);for(;t.c!=0;)for(i=0;i<t.j.g;i++)t.e[i]&&(n.Ie(t,i),n.Ke(t));for(e&&n.Ue(t,e),t.b-t.k>=2&&n.Ce(t),r=0;r<t.j.f;r++)t.d[r]&&n.op(t.j,r)!=0&&(e&&(n.Bl(t.j,r)==1&&n.Hm(t.j,r)||n.Bl(t.j,r)==-1&&n.Tl(t.j,r)==5)?n.jn(t.j,r,0):n.wn(t.j,r,32),++t.k);return t.b==t.k;},n.He=function(t,A){return t==A-1?0:t+1;},n.Ie=function(t,A){var e,r,P,i;for(n.im(t.j,A)==1&&(n.Mn(t.j,A,2),t.k+=2),P=0;P<2;P++)for(e=n.Zl(t.j,P,A),t.d[e]=false,i=0;i<n.$o(t.j,e);i++)r=n._o(t.j,e,i),t.e[r]&&(t.e[r]=false,--t.c);},n.Je=function(t){var A,e,r,P,i,D,u,Q;for(e=0;e<t.j.g;e++)if(t.e[e])for(P=0;P<2;P++){for(u=n.Zl(t.j,P,e),A=false,D=0;D<n.$o(t.j,u);D++)if(e!=n._o(t.j,u,D)&&t.e[n._o(t.j,u,D)]){A=true;break;}if(!A){for(Q=e,r=n.Zl(t.j,1-P,e);Q!=-1;)for(t.d[u]=false,t.e[Q]=false,--t.c,n.Mn(t.j,Q,8),Q=-1,u=r,i=0;i<n.$o(t.j,u);i++)if(t.e[n._o(t.j,u,i)])if(Q==-1)Q=n._o(t.j,u,i),r=n.Zo(t.j,u,i);else {u=-1,Q=-1;break;}u!=-1&&(t.d[r]=false);break;}}},n.Ke=function(t){var A,e,r,P,i,D,u;do for(u=false,e=0;e<t.j.g;e++)if(t.e[e]){for(i=false,P=0;P<2;P++){for(r=n.Zl(t.j,P,e),A=false,D=0;D<n.$o(t.j,r);D++)if(e!=n._o(t.j,r,D)&&t.e[n._o(t.j,r,D)]){A=true;break;}if(!A){i=true;break;}}i&&(u=true,n.Ie(t,e));}while(u);},n.Le=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(b=0;b<A.j.a.length;b++)if(U.xj(A.k,b).length!=6&&t.i[b]){for(y=true,k=U.xj(A.k,b),L=0;L<k.length;L++)if(!t.e[k[L]]){y=false;break;}if(y){for(u=-1,Q=0,g=U.xj(A.j,b),l=0;l<g.length;l++)r=g[l],c=t.g[r]?n.ye(t,r,false):n.ze(t,r,false),Q<c&&(Q=c,u=l);if(u!=-1){if(T=g[u],e){for(P=g,i=0,D=P.length;i<D;++i)r=P[i],r!=T&&n.Be(t,r,true);t.g[T]?n.ye(t,T,true):n.ze(t,T,true);}return n.Pe(t,T),t.g[T]&&n.Se(t,g,k,u),true;}}}return  false;},n.Me=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;for(S=B(o.Q7,H,6,t.j.g,15,1),g=0;g<A.j.a.length;g++){for(v=U.xj(A.k,g),y=true,Q=0;Q<v.length;Q++)if(!t.e[v[Q]]){y=false;break;}if(y)for(u=0;u<v.length;u++)++S[v[u]];}for(P=t.c,b=0;b<A.j.a.length;b++)for(D=false,k=U.xj(A.j,b),v=U.xj(A.k,b),u=0;u<v.length&&!D;u++)if(S[v[u]]>1&&(i=n.He(u,v.length),S[v[i]]==1)){for(l=false,c=n.He(i,v.length);S[v[c]]==1;)(!n.Be(t,k[c],false)||v.length!=6&&n.Tl(t.j,k[c])!=6)&&(l=true),c=n.He(c,v.length);if(!l&&(r=c>i?c-i:c+v.length-i,r>2&&(r&1)==1)){for(T=1;T<r;T+=2)L=i+T<v.length?i+T:i+T-v.length,e&&(n.Be(t,k[L],true),n.Be(t,k[L==k.length-1?0:L+1],true)),n.Ie(t,v[L]);D=true;}}return P!=t.c;},n.Ne=function(t,A,e){var r,P,i,D,u,Q,l,L;for(r=false,Q=0;Q<A.j.a.length;Q++)if(U.xj(A.k,Q).length==6){for(u=true,l=U.xj(A.j,Q),L=U.xj(A.k,Q),i=0;i<6;i++)if(!n.Be(t,l[i],false)||!t.e[L[i]]){u=false;break;}if(u){if(e)for(D=0;D<6;D++)n.Be(t,l[D],true);for(P=0;P<6;P+=2)n.Ie(t,L[P]);r=true;break;}}return r;},n.Oe=function(t,A){var e;for(e=0;e<t.j.g;e++)t.e[e]&&n.wt(A,e)&&(n.Pe(t,n.Zl(t.j,0,e)),n.Pe(t,n.Zl(t.j,1,e)));},n.Pe=function(t,A){var e,r;for(t.d[A]&&(t.d[A]=false,--t.b),r=0;r<n.$o(t.j,A);r++)e=n._o(t.j,A,r),t.e[e]&&(t.e[e]=false,--t.c);},n.Qe=function(t){var A,e,r,P,i;for(e=0;e<t.j.g;e++)if(n.fm(t.j,e)==2){for(P=0;P<2;P++)if(A=n.Zl(t.j,P,e),n.Tl(t.j,A)<=8){for(i=0;i<n.$o(t.j,A);i++)if(r=n._o(t.j,A,i),t.e[r]){n.Pe(t,A);break;}}}},n.Re=function(t,A){var e;for(e=0;e<t.j.f;e++)t.d[e]&&n.pp(t.j,e)==0&&(!A||n.Tl(t.j,e)==5&&n.Bl(t.j,e)<0||n.Tl(t.j,e)==6||n.Tl(t.j,e)==14||n.Hm(t.j,e)&&n.Bl(t.j,e)>0)&&n.Pe(t,e);},n.Se=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k;if(y=A[r==0?4:r-1],k=A[r==4?0:r+1],n.$o(t.j,y)==3&&n.$o(t.j,k)==3){for(c=B(o.Q7,H,6,t.j.f,15,1),T=B(o.vfb,Tt,6,t.j.g,16,1),i=e,D=0,u=i.length;D<u;++D)Q=i[D],T[Q]=true;for(P=0;P<t.j.g;P++)t.e[P]||(T[P]=true);if(b=n.yp(t.j,c,y,k,14,null,T),b==14){for(g=true,L=2;L<=12;L++)if((L&3)!=1^n.Uo(t.j,c[L])==5){g=false;break;}if(g)for(n.Pe(t,c[7]),l=1;l<=5;l+=2)n.Ie(t,n.Xo(t.j,c[l],c[l+1])),n.Ie(t,n.Xo(t.j,c[l+7],c[l+8]));}}},n.Te=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z;for(p=B(o.vfb,Tt,6,t.j.f,16,1),O=0;O<A.j.a.length;O++)if(tt=U.xj(A.k,O).length,(tt==3||tt==5||tt==6||tt==7)&&t.i[O])for(r=U.xj(A.j,O),P=0,i=r.length;P<i;++P)e=r[P],p[e]=true;for(T=B(o.Q7,H,6,t.j.f,15,1),b=B(o.vfb,Tt,6,t.j.f,16,1),l=0;l<t.j.g;l++)D=n.Zl(t.j,0,l),u=n.Zl(t.j,1,l),!p[D]&&!p[u]&&(t.e[l]&&(++T[D],++T[u]),n.im(t.j,l)==16&&(b[D]=true,b[u]=true));for(K=B(o.Q7,H,6,t.j.f,15,1),c=B(o.Q7,H,6,t.j.f,15,1),z=0;z<t.j.f;z++)if(T[z]==1)for(c[0]=z,y=0,g=0;y<=g;){for(k=0;k<n.$o(t.j,c[y]);k++)if(t.e[n._o(t.j,c[y],k)]&&(L=n.Zo(t.j,c[y],k),(y==0||L!=c[y-1])&&T[L]!=0)){if(c[++g]=L,(T[L]&1)!=0){for(q=1;q<g;q+=2)K[c[q]]=-1;g=0;}break;}++y;}for(Q=B(o.vfb,Tt,6,t.j.f,16,1),Qt=0;Qt<t.j.f;Qt++)if(!Q[Qt]&&T[Qt]!=0){for(c[0]=Qt,Q[Qt]=true,y=0,g=0;y<=g;){for(k=0;k<n.$o(t.j,c[y]);k++)t.e[n._o(t.j,c[y],k)]&&(L=n.Zo(t.j,c[y],k),Q[L]||(c[++g]=L,Q[L]=true));++y;}if((g&1)==0){for(v=0;v<=g;v++)K[c[v]]==0&&(K[c[v]]=n.Ae(t,c[v],false));for(E=true,S=0;S<=g;S++)if(K[c[S]]<=0&&!n.Be(t,c[S],false)){E=false;break;}if(E){for(G=0,V=-1,k=0;k<=g;k++)G<K[c[k]]&&(G=K[c[k]],V=c[k]);G>0&&(n.Ae(t,V,true),n.Pe(t,V));}}}},n.Ue=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g;for(e=0;e<t.j.f;e++)if(n.bw(t.j,1),n.Tl(t.j,e)==7&&n.Bl(t.j,e)==1&&n.Qo(t.j,e)==1&&!n.Fe(t,e,A))for(u=false,Q=B(o.Q7,H,6,t.j.f,15,1),g=B(o.Q7,H,6,t.j.f,15,1),l=B(o.Q7,H,6,t.j.f,15,1),Q[0]=e,g[e]=-1,l[e]=1,i=0,L=0;i<=L&&!u;){for(D=Q[i],y=0;y<n.$o(t.j,D)&&!u;y++)if(T=(l[D]&1)==0^n.fm(t.j,n._o(t.j,D,y))>1,P=n.Zo(t.j,D,y),l[P]==0&&T)if(n.Tl(t.j,P)==7&&n.Qo(t.j,P)==0&&n.$o(t.j,P)==2){if((l[D]&1)==0){for(n.jn(t.j,e,0),b=D,c=0;c<l[D];c++)r=n.Xo(t.j,P,b),n.In(t.j,r,n.fm(t.j,r)==1?2:n.fm(t.j,r)-1),P=b,b=g[P];u=true;}}else Q[++L]=P,g[P]=D,l[P]=l[D]+1;++i;}},n.Ve=function(t){n.We.call(this,t);},n.We=function(t){this.j=t;},x(89,1,{},n.Ve),s.a=false,s.b=0,s.c=0,s.k=0,o.X7=N(89),n.Xe=function(t,A){var e,r;for(e=0,r=0;r<t.j[A];r++)t.n[A][r]==2&&(n.Tl(t,t.i[A][r])==7||n.Tl(t,t.i[A][r])==8||n.Tl(t,t.i[A][r])==16)&&++e;return e;},n.Ye=function(t,A){var e,r,P,i,D,u;if(t.s[A]==0)return  false;for(u=true,e=t.s[A],i=t.j[A],D=0,r=0;r<i;r++)P=t.i[A][r],D+=t.s[P];return d.Math.abs(e)<=d.Math.abs(D)&&h.Jjb(e)!=h.Jjb(D)&&(u=false),u;},n.Ze=function(t,A){var e,r,P,i,D,u,Q;if(t.C[A]!=8||t.s[A]!=0||t.j[A]!=1||t.n[A][0]!=1)return  false;if(e=t.i[A][0],t.C[e]==6){for(D=t.j[e],r=0;r<D;r++)if(P=t.i[e][r],P!=A&&t.C[P]==8&&(i=n.Xo(t,e,P),t.H[i]==2))return  true;}else if(t.C[e]==7){if(t.s[e]==1)return  true;}else if(t.C[e]==16){for(u=t.j[e],Q=0,r=0;r<u;r++)P=t.i[e][r],P!=A&&t.C[P]==8&&(i=n.Xo(t,e,P),t.H[i]==2&&++Q);if(Q==2)return  true;}else if(n.$e(t,A))return  true;return  false;},n.$e=function(t,A){var e,r,P,i,D;if(t.C[A]!=8||t.j[A]!=1)return  false;if(e=t.i[A][0],t.C[e]==15){for(D=t.j[e],r=0;r<D;r++)if(P=t.i[e][r],P!=A&&t.C[P]==8&&(i=n.Xo(t,e,P),t.H[i]==2))return  true;}return  false;},n._e=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G;if(t.C[A]!=7||t.s[A]!=0||t.j[A]+t.o[A]>3)return  false;if(A<t.f&&n.ot(t.p,A)){if(t.o[A]!=1||n.To(t,A,7)!=1)return  false;for(E=(n.bw(t,7),t.p),S=0;S<E.j.a.length;S++)if(n.qt(E,S,A)){if(U.xj(E.k,S).length==5||U.xj(E.k,S).length==6){for(q=U.xj(E.j,S),k=-1,Q=0;Q<q.length;Q++)if(q[Q]==A){k=Q;break;}for(P=0,v=null,g=null,q.length==5&&(v=B(o.Q7,H,6,2,15,1),v[0]=q[k-1<0?k+4:k-1],v[1]=q[k-4<0?k+1:k-4],g=B(o.Q7,H,6,2,15,1),g[0]=q[k-2<0?k+3:k-2],g[1]=q[k-3<0?k+2:k-3]),q.length==6&&(v=B(o.Q7,H,6,3,15,1),v[0]=q[k-1<0?k+5:k-1],v[1]=q[k-3<0?k+3:k-3],v[2]=q[k-5<0?k+1:k-5],g=B(o.Q7,H,6,2,15,1),g[0]=q[k-2<0?k+4:k-2],g[1]=q[k-4<0?k+2:k-4]),l=0;l<q.length;l++)A!=q[l]&&n.Tl(t,q[l])==7&&n.Qo(t,q[l])==1&&--P;for(L=0;L<v.length;L++){for(i=-1,D=-1,b=0;b<n.$o(t,v[L]);b++)if(!n.Up(t,n._o(t,v[L],b))){i=n.Zo(t,v[L],b),D=n._o(t,v[L],b);break;}if(i!=-1){if(t.C[i]==7&&t.o[i]==0&&t.j[i]+t.o[i]<=3&&!n.af(t,i,false)){++P;continue;}if(t.C[i]==8&&t.j[i]==1){P+=2;continue;}if(D<t.g&&n.pt(t.p,D)){for(V=0;V<E.j.a.length;V++)if(E.e[V]&&n.qt(E,V,i)){for(p=U.xj(E.j,V),c=0;c<p.length;c++)if(n.Tl(t,p[c])==7&&n.Qo(t,p[c])==1){--P;break;}break;}}}}for(y=0;y<g.length;y++){for(i=-1,c=0;c<n.$o(t,g[y]);c++)n.Up(t,n._o(t,g[y],c))||(i=n.Zo(t,g[y],c));n.Tl(t,g[y])==6?i!=-1&&n.Xe(t,i)!=0&&--P:n.Tl(t,g[y])==7&&n.Qo(t,g[y])==0&&(i==-1||!(i<t.f&&n.ot(t.p,i))&&n.Xe(t,i)==0)&&++P;}return P>0;}break;}return  false;}if(t.o[A]>1)return  false;if(t.o[A]==1){for(T=-1,G=0,Q=0;Q<t.j[A];Q++){if(r=t.i[A][Q],t.n[A][Q]==2){if(t.C[r]!=6)return  false;T=r;continue;}if(t.C[r]==8)return  false;if(t.C[r]==7){--G,n.af(t,r,false)&&--G;continue;}r<t.f&&n.ot(t.p,r)&&--G;}if(T==-1)return  false;for(e=0,l=0;l<t.j[T];l++)if(t.n[T][l]==1){if(r=t.i[T][l],n.Xe(t,r)!=0)return  false;r<t.f&&n.ot(t.p,r)&&++e,t.C[r]==7&&!n.af(t,r,true)&&++G,(t.C[r]==8||t.C[r]==16)&&--G;}return e==2&&--G,G>=0;}for(u=0;u<t.j[A];u++)if(r=t.i[A][u],r<t.f&&n.ot(t.p,r)||t.C[r]!=6||n.Xe(t,r)!=0||t.o[r]!=0&&n.bf(t,r))return  false;return  true;},n.af=function(t,A,e){var r,P,i;for(r=false,i=0;i<t.j[A];i++)if(!n.Up(t,t.k[A][i])&&t.n[A][i]==1&&(P=t.i[A][i],!(P<t.f&&n.ot(t.p,P))&&(t.C[P]==6&&n.Xe(t,P)==1||t.C[P]==16&&n.Xe(t,P)==2))){if(r||!e)return  true;r=true;}return  false;},n.bf=function(t,A){var e,r,P;for(r=0;r<t.j[A];r++)if(t.n[A][r]!=1){for(e=t.i[A][r],P=0;P<t.j[e];P++)if(t.n[e][P]==1&&n.Xe(t,t.i[e][P])!=0)return  true;}return  false;},n.ef=function(){n.ef=f1,n.cf=j(F(o.ufb,1),Yt,6,15,[-1,-1,-1,0,0,1,2,3,4,5,-1,0,0,0,6,7,8,9,-1,0,0,10,10,10,10,10,10,10,10,10,10,1,11,11,12,13,-1,0,0,10,10,10,10,10,10,10,10,10,10,0,0,0,11,14,-1,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,10,10,10,10,10,10,10,10,1,1,1,1,-1,-1,-1,-1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]),n.df=j(F(o.ufb,1),Yt,6,15,[-1,-1,-1,0,0,0,2,5,5,5,-1,0,0,0,0,9,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);},n.ff=function(t,A,e){n.ef();var r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(n.bw(t,7),c=B(o.R7,e0,6,t.j[A],14,1),T=0,Q=0;Q<t.j[A];Q++)if(P=t.i[A][Q],t.C[P]!=1){if(D=0,(e&32)!=0&&(i=Ct(t.n[A][Q]),e==32190?nt(i,3)<0&&n.Yp(t,t.k[A][Q])&&t.o[A]==1&&(i=0):nt(i,3)<0&&n.Up(t,t.k[A][Q])&&(i=0),D=Pt(D,h1(i,4))),(e&128)!=0){if(n.cf[t.C[P]]==-1)throw st(new h.cC(fP+t.C[P]));D=v1(D,Ct(n.cf[t.C[P]]));}else if((e&64)!=0){if(n.df[t.C[P]]==-1)throw st(new h.cC(fP+t.C[P]));D=v1(D,Ct(n.df[t.C[P]]));}for((e&256)!=0&&(b=t.j[P]-1,b>3&&(b=3),(e&512)==0&&b>1&&(b=1),D=Pt(D,o.Kfb(b<<6))),(e&f0)!=0&&(t.u[P]&8)!=0&&(D=Pt(D,256)),(e&B1)!=0&&P<t.f&&n.ot(t.p,P)&&(D=Pt(D,512)),(e&N0)!=0&&(t.u[P]&L0)!=0&&(D=Pt(D,f0)),L=0;nt(D,c[L])<0;)++L;for(y=Q;y>L;y--)c[y]=c[y-1];c[L]=D,++T;}for(T>4&&(T=4),r=0,l=0;l<T;l++)r=h1(r,11),r=Pt(r,c[l]);if(r=h1(r,15),n.cf[t.C[A]]==-1)throw st(new h.cC(fP+t.C[A]));if(r=Pt(r,Ct(n.cf[t.C[A]])),(e&2)!=0?(k=t.p&&A<t.f?n.dt(t.p,A):0,k>9&&(k=9),k>2&&(k-=2),r=Pt(r,o.Kfb(k<<4))):(e&1)!=0&&(t.u[A]&8)!=0&&(r=Pt(r,64)),(e&4)!=0&&A<t.f&&n.ot(t.p,A)&&(r=Pt(r,f0)),(e&8)!=0&&(t.u[A]&Z1)!=0&&(r=Pt(r,B1)),(e&16)!=0&&(t.u[A]&L0)!=0&&(r=Pt(r,Z1)),(e&Z1)!=0&&(n.Ye(t,A)&&(r=Pt(r,L0)),n._e(t,A))){for(u=0;u<t.f;u++)if(n.Ze(t,u)){r=Pt(r,N0);break;}}return (e&L0)!=0&&(g=Ct(n.To(t,A,10)),r=Pt(r,h1(g,7))),r;},n.gf=function(t,A,e,r,P,i){var D,u,Q,l;for(Q=0,u=0;u<t.R.f;u++)pt(J(n.Ol(t.R,t.B[u]),r),0)&&++Q;if(Q!=0)for(n.eg(t,A),n._f(t,o.Kfb(Q),e),D=0;D<t.R.f;D++)l=J(n.Ol(t.R,t.B[D]),r),nt(l,0)!=0&&(n._f(t,o.Kfb(D),e),P!=1&&n._f(t,A0(l,i),P));},n.hf=function(t,A,e,r,P,i){var D,u,Q,l;for(Q=0,u=0;u<t.R.g;u++)(n.hm(t.R,t.C[u])&r)!=0&&++Q;if(Q!=0)for(n.eg(t,A),n._f(t,o.Kfb(Q),e),D=0;D<t.R.g;D++)l=n.hm(t.R,t.C[D])&r,l!=0&&(n._f(t,o.Kfb(D),e),P!=1&&n._f(t,o.Kfb(l>>i),P));},n.jf=function(t){for(var A,e,r,P,i,D;;){for(i=B(o.vfb,Tt,6,t.U+1,16,1),r=-1,e=0;e<t.R.f;e++)n.pp(t.R,e)!=0&&(i[t.f[e]]&&r<t.f[e]&&(r=t.f[e]),i[t.f[e]]=true);if(r==-1)break;for(P=0,A=0;A<t.R.f;A++)D=0,t.f[A]==r&&(D=++P),n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),n.ah(t.e[A],8,o.Kfb(D));t.U=n.Kf(t);}},n.kf=function(t){var A,e,r,P;for(e=0;e<t.R.f;e++)n.eh(t.e[e],e),n.ah(t.e[e],2*t.c+4,h1(Ct(t.f[e]),t.c+4));for(P=false,A=0;A<t.R.f;A++)P=P|n.xf(t,A,3);for(r=0;r<t.R.g;r++)P=P|n.sf(t,r,3);if(!P)return  false;for(;t.U<t.R.f&&(P=n.Hf(t),!!P);)n.Jf(t),t.P&&n.uh(t.P,t.f);return  true;},n.lf=function(t){var A,e,r,P,i;for(e=0;e<t.R.f;e++)n.eh(t.e[e],e),n.ah(t.e[e],t.c+1,O0(2,Ct(t.f[e])));for(i=B(o.Q7,H,6,t.U+1,15,1),r=0;r<t.R.f;r++)++i[t.f[r]];for(P=1;i[P]==1;)++P;for(A=0;A<t.R.f;A++)if(t.f[A]==P){n.bh(t.e[A],1);break;}t.U=n.Kf(t);},n.mf=function(t,A,e){var r,P,i,D,u;return n.Tl(t.R,A)!=6&&n.Tl(t.R,A)!=7||(P=n.Zo(t.R,A,0),i=n.Zo(t.R,A,1),n.Qo(t.R,P)!=1||n.Qo(t.R,i)!=1)||n.$o(t.R,P)==1||n.$o(t.R,i)==1||n.Lo(t.R,P)>3||n.Lo(t.R,i)>3||(D=new n.lk(t.R,t.f,A,P),D.f&&e==1)||(u=new n.lk(t.R,t.f,A,i),u.f&&e==1)||D.f&&u.f?false:(e==3&&(D.f&&D.c&&(t.W[A]=true),u.f&&u.c&&(t.W[A]=true)),r=t.jb?n.of(t,D,u):n.nf(D,u),e==1?t.db[A]=r:e==2&&(D.f&&(r==1?n.bh(t.e[D.b],Ct(t.f[P])):n.bh(t.e[D.d],Ct(t.f[P]))),u.f&&(r==2?n.bh(t.e[u.b],Ct(t.f[i])):n.bh(t.e[u.d],Ct(t.f[i])))),true);},n.nf=function(t,A){var e,r,P;if(r=n.kk(t),P=n.kk(A),r==-1||P==-1||(r+P&1)==0)return 3;switch(e=0,r+P){case 3:case 7:e=2;break;case 5:e=1;}return e;},n.of=function(t,A,e){var r,P;return r=B(o.Q7,H,6,4,15,1),r[0]=A.b,r[1]=A.a,r[2]=e.a,r[3]=e.b,P=n.el(t.R,r),d.Math.abs(P)<.3||d.Math.abs(P)>G8?3:P<0?2:1;},n.pf=function(t,A,e){var r,P,i,D,u;return !n.Vp(t.R,A)||(r=n.Zl(t.R,0,A),P=n.Zl(t.R,1,A),D=new n.lk(t.R,t.f,r,P),D.f&&e==1)||(u=new n.lk(t.R,t.f,P,r),u.f&&e==1)||D.f&&u.f?false:(e==3&&(D.f&&(t.V[A]=n.xg(t,P)),u.f&&(t.V[A]=n.xg(t,r))),i=t.jb?n.rf(t,D,u):n.qf(D,u),e==1?t.o[A]=i:e==2&&(D.f&&(i==2?n.bh(t.e[D.b],Ct(t.f[P])):n.bh(t.e[D.d],Ct(t.f[P]))),u.f&&(i==2?n.bh(t.e[u.b],Ct(t.f[r])):n.bh(t.e[u.d],Ct(t.f[r])))),true);},n.qf=function(t,A){var e,r,P;if(r=n.kk(t),P=n.kk(A),r==-1||P==-1||(r+P&1)==0)return 3;switch(e=0,r+P){case 3:case 7:e=1;break;case 5:e=2;}return e;},n.rf=function(t,A,e){var r,P;return r=B(o.Q7,H,6,4,15,1),r[0]=A.b,r[1]=A.a,r[2]=e.a,r[3]=e.b,P=n.el(t.R,r),d.Math.abs(P)<.3||d.Math.abs(P)>G8?3:P<0?1:2;},n.sf=function(t,A,e){var r,P,i,D,u,Q,l,L;if(t.o[A]!=0)return  false;if(t.R.K){for(l=0;l<2;l++)for(r=n.Zl(t.R,l,A),L=0;L<n.Lo(t.R,r);L++)if(n.Jm(t.R,n.Zo(t.R,r,L)))return  false;}return n.fm(t.R,A)==1?n.pf(t,A,e):n.fm(t.R,A)!=2||n.Up(t.R,A)||(i=n.Zl(t.R,0,A),D=n.Zl(t.R,1,A),n.$o(t.R,i)==1||n.$o(t.R,D)==1)||n.$o(t.R,i)>3||n.$o(t.R,D)>3||n.Qo(t.R,i)==2||n.Qo(t.R,D)==2||(u=new n.lk(t.R,t.f,D,i),u.f&&e==1)||(Q=new n.lk(t.R,t.f,i,D),Q.f&&e==1)||u.f&&Q.f?false:(e==3&&(u.f&&u.c&&(t.V[A]=true),Q.f&&Q.c&&(t.V[A]=true)),P=n.Gm(t.R,A)?3:t.jb?n.uf(t,u,Q):n.tf(u,Q),e==1?t.o[A]=P:e==2&&(u.f&&(P==1?n.bh(t.e[u.b],Ct(t.f[i])):P==2&&n.bh(t.e[u.d],Ct(t.f[i]))),Q.f&&(P==1?n.bh(t.e[Q.b],Ct(t.f[D])):P==2&&n.bh(t.e[Q.d],Ct(t.f[D])))),true);},n.tf=function(t,A){return n.kk(t)==-1||n.kk(A)==-1||((n.kk(t)|n.kk(A))&1)!=0?3:n.kk(t)==n.kk(A)?1:2;},n.uf=function(t,A,e){var r,P,i,D,u,Q,l;return i=B(o.O7,Ft,6,3,15,1),i[0]=n.Ql(t.R,e.a)-n.Ql(t.R,A.a),i[1]=n.Rl(t.R,e.a)-n.Rl(t.R,A.a),i[2]=n.Sl(t.R,e.a)-n.Sl(t.R,A.a),Q=B(o.O7,Ft,6,3,15,1),Q[0]=n.Ql(t.R,A.b)-n.Ql(t.R,A.a),Q[1]=n.Rl(t.R,A.b)-n.Rl(t.R,A.a),Q[2]=n.Sl(t.R,A.b)-n.Sl(t.R,A.a),l=B(o.O7,Ft,6,3,15,1),l[0]=n.Ql(t.R,e.b)-n.Ql(t.R,e.a),l[1]=n.Rl(t.R,e.b)-n.Rl(t.R,e.a),l[2]=n.Sl(t.R,e.b)-n.Sl(t.R,e.a),D=B(o.O7,Ft,6,3,15,1),D[0]=i[1]*Q[2]-i[2]*Q[1],D[1]=i[2]*Q[0]-i[0]*Q[2],D[2]=i[0]*Q[1]-i[1]*Q[0],u=B(o.O7,Ft,6,3,15,1),u[0]=i[1]*D[2]-i[2]*D[1],u[1]=i[2]*D[0]-i[0]*D[2],u[2]=i[0]*D[1]-i[1]*D[0],r=(Q[0]*u[0]+Q[1]*u[1]+Q[2]*u[2])/(d.Math.sqrt(Q[0]*Q[0]+Q[1]*Q[1]+Q[2]*Q[2])*d.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),P=(l[0]*u[0]+l[1]*u[1]+l[2]*u[2])/(d.Math.sqrt(l[0]*l[0]+l[1]*l[1]+l[2]*l[2])*d.Math.sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2])),r<0^P<0?1:2;},n.vf=function(t,A){var e,r,P,i;return e=n.zl(t.R,A),r=n.mp(t.R,A,false),P=n.mp(t.R,A,true),i=-1,r!=P?e!=-1&&e>r?i=e<<24>>24:i=r<<24>>24:e!=-1?(e>P||e<P&&e>=n.vp(t.R,A))&&(i=e<<24>>24):!n.Aq(t.R,A)&&n.cp(t.R,A)!=0&&(i=n.vp(t.R,A),i-=n.mm(t.R,A,i)),n.Rf(t,A,i),i;},n.wf=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(r=B(o.Q7,H,6,t.O,15,1),A=0;A<t.R.f;A++){for(L=n.$o(t.R,A)+n.rp(t.R,A),l=0,i=0;i<n.Mo(t.R,A);i++)if(i<n.$o(t.R,A)||i>=n.Lo(t.R,A)){for(y=2*t.f[n.Zo(t.R,A,i)],e=n._o(t.R,A,i),n.fm(t.R,e)==2&&(n.Up(t.R,e)||++y),u=0;u<l&&!(y<r[u]);u++);for(Q=l;Q>u;Q--)r[Q]=r[Q-1];r[u]=y,++l;}for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),D=L;D<t.O;D++)n.ah(t.e[A],t.c+1,0);for(P=0;P<L;P++)n.ah(t.e[A],t.c+1,o.Kfb(r[P]));}},n.xf=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g;if(t.db[A]!=0||n.Tl(t.R,A)!=5&&n.Tl(t.R,A)!=6&&n.Tl(t.R,A)!=7&&n.Tl(t.R,A)!=14&&n.Tl(t.R,A)!=15&&n.Tl(t.R,A)!=16||(t.Q&Z1)!=0&&n.Tl(t.R,A)>=14)return  false;if(n.Qo(t.R,A)!=0){if(n.Wp(t.R,A))return n.mf(t,A,e);if(n.Tl(t.R,A)!=15&&n.Tl(t.R,A)!=16)return  false;}if(n.$o(t.R,A)<3||n.Lo(t.R,A)>4||n.Bl(t.R,A)>0&&n.Tl(t.R,A)==6||n.Tl(t.R,A)==5&&n.Lo(t.R,A)!=4)return  false;if(t.R.K){if(n.Jm(t.R,A))return  false;for(u=0;u<n.Lo(t.R,A);u++)if(n.Jm(t.R,n.Zo(t.R,A,u)))return  false;}if(n.Tl(t.R,A)==7&&!t.S[A])return  false;for(b=B(o.Q7,H,6,4,15,1),g=B(o.Q7,H,6,4,15,1),L=B(o.vfb,Tt,6,4,16,1),Q=0;Q<n.Lo(t.R,A);Q++){for(i=-1,P=0,l=0;l<n.Lo(t.R,A);l++)L[l]||i<t.f[n.Zo(t.R,A,l)]&&(i=t.f[n.Zo(t.R,A,l)],P=l);b[Q]=P,g[Q]=i,L[P]=true;}if(n.Lo(t.R,A)==4&&g[0]==g[1]&&g[2]==g[3]||n.Lo(t.R,A)==4&&(g[0]==g[2]||g[1]==g[3])||n.Lo(t.R,A)==3&&g[0]==g[2])return  false;for(y=0,T=0,c=false,D=1;D<n.Lo(t.R,A);D++)if(g[D-1]==g[D]){if(e==1||g[D]==0)return  false;y=n.Zo(t.R,A,b[D-1]),T=n.Zo(t.R,A,b[D]),e==3&&n.gq(t.R,n._o(t.R,A,b[D]))&&(t.W[A]=true),c=true;}return e!=1&&!c?false:(r=t.jb?n.zf(t,A,b):n.yf(t,A,b),e==1?t.db[A]=r:e==2&&(r==1?n.bh(t.e[y],Ct(t.f[A])):r==2&&n.bh(t.e[T],Ct(t.f[A]))),true);},n.yf=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T;for(T=j(F(o.Q7,2),E1,7,0,[j(F(o.Q7,1),H,6,15,[2,1,2,1]),j(F(o.Q7,1),H,6,15,[1,2,2,1]),j(F(o.Q7,1),H,6,15,[1,1,2,2]),j(F(o.Q7,1),H,6,15,[2,1,1,2]),j(F(o.Q7,1),H,6,15,[2,2,1,1]),j(F(o.Q7,1),H,6,15,[1,2,1,2])]),r=B(o.O7,Ft,6,n.Lo(t.R,A),15,1),D=0;D<n.Lo(t.R,A);D++)r[D]=n.Yl(t.R,n.Zo(t.R,A,e[D]),A);if(l=n.ep(t.R,A,e,r,null)<<24>>24,l!=3)return l;for(L=0,y=0,u=0;u<n.Lo(t.R,A);u++)P=n._o(t.R,A,e[u]),n.Zl(t.R,0,P)==A&&(n.im(t.R,P)==129&&(y!=0&&n.Vn(t.R,A),L=u,y=1),n.im(t.R,P)==257&&(y!=0&&n.Vn(t.R,A),L=u,y=2));if(y==0)return 3;for(i=1;i<n.Lo(t.R,A);i++)r[i]<r[0]&&(r[i]+=N1);if(n.Lo(t.R,A)==3){switch(L){case 0:(r[1]<r[2]&&r[2]-r[1]<A1||r[1]>r[2]&&r[1]-r[2]>A1)&&(y=3-y);break;case 1:r[2]-r[0]>A1&&(y=3-y);break;case 2:r[1]-r[0]<A1&&(y=3-y);}return y==1?2:1;}return Q=0,r[1]<=r[2]&&r[2]<=r[3]?Q=0:r[1]<=r[3]&&r[3]<=r[2]?Q=1:r[2]<=r[1]&&r[1]<=r[3]?Q=2:r[2]<=r[3]&&r[3]<=r[1]?Q=3:r[3]<=r[1]&&r[1]<=r[2]?Q=4:r[3]<=r[2]&&r[2]<=r[1]&&(Q=5),T[Q][L]==y?2:1;},n.zf=function(t,A,e){var r,P,i,D,u,Q;for(r=B(o.Q7,H,6,4,15,1),u=0;u<n.Lo(t.R,A);u++)r[u]=n.Zo(t.R,A,e[u]);for(n.Lo(t.R,A)==3&&(r[3]=A),P=U1(o.O7,[o1,Ft],[15,6],15,[3,3],2),D=0;D<3;D++)P[D][0]=n.Ql(t.R,r[D+1])-n.Ql(t.R,r[0]),P[D][1]=n.Rl(t.R,r[D+1])-n.Rl(t.R,r[0]),P[D][2]=n.Sl(t.R,r[D+1])-n.Sl(t.R,r[0]);return Q=B(o.O7,Ft,6,3,15,1),Q[0]=P[0][1]*P[1][2]-P[0][2]*P[1][1],Q[1]=P[0][2]*P[1][0]-P[0][0]*P[1][2],Q[2]=P[0][0]*P[1][1]-P[0][1]*P[1][0],i=(P[2][0]*Q[0]+P[2][1]*Q[1]+P[2][2]*Q[2])/(d.Math.sqrt(P[2][0]*P[2][0]+P[2][1]*P[2][1]+P[2][2]*P[2][2])*d.Math.sqrt(Q[0]*Q[0]+Q[1]*Q[1]+Q[2]*Q[2])),i>0?1:2;},n.Af=function(t){var A,e;for(A=0,U.Trb(t.e),e=0;e<t.e.length;e++)(e==0||n.dh(t.e[e],t.e[e-1])!=0)&&++A,t.f[t.e[e].a]=A;return A;},n.Bf=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T;if(!t.A){for(t.A=new U.Mj(),L=0,y=B(o.Q7,H,6,t.R.f,15,1),D=B(o.Q7,H,6,t.R.f,15,1),Q=B(o.Q7,H,6,t.R.g,15,1),A=0;A<t.R.f;A++)if(y[A]==0&&(n.fq(t.R,A)||n.Qo(t.R,A)==1)){for(D[0]=A,u=1,l=0,y[A]=++L,e=B(o.vfb,Tt,6,t.R.g,16,1),i=0;i<u;i++)for(T=0;T<n.$o(t.R,D[i]);T++)P=n._o(t.R,D[i],T),(n.gq(t.R,P)||n.fm(t.R,P)==2||n.Vp(t.R,P))&&(r=n.Zo(t.R,D[i],T),e[P]||(Q[l++]=P,e[P]=true),y[r]==0&&(D[u++]=r,y[r]=L));U.sj(t.A,new n.kh(D,u,Q,l));}}},n.Cf=function(t){var A;for(t.S=B(o.vfb,Tt,6,t.R.f,16,1),A=0;A<t.R.f;A++)if(n.Tl(t.R,A)==7){if(n.$o(t.R,A)==4){t.S[A]=true;continue;}if(n.$o(t.R,A)==3){if(n.Uo(t.R,A)==3){t.S[A]=true;continue;}if(n.Bl(t.R,A)==1){t.S[A]=true;continue;}if(n.$p(t.R,A,true))continue;if((t.Q&32)!=0){t.S[A]=true;continue;}if(n.eq(t.R,A)){t.S[A]=true;continue;}}}},n.Df=function(t,A){var e,r,P,i;for(P=false,r=0;r<t.R.g;r++)n.sf(t,r,1)&&(t.q[r]=t.L,A&&n.Sf(t,r),P=true);for(i=false,e=0;e<t.R.f;e++)n.xf(t,e,1)&&(t.ib[e]=t.L,A&&n.Tf(t,e),i=true);return i&&(t.L=!t.L),P||i;},n.Ef=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K;for(S=B(o.vfb,Tt,6,t.R.f,16,1),p=B(o.vfb,Tt,6,t.R.g,16,1),A=0,q=false,(t.Q&128)!=0&&(t.Y=B(o.Q7,H,6,t.R.f,15,1),t.X=B(o.Q7,H,6,t.R.g,15,1)),r=0;r<t.R.f;r++)t.W[r]&&(t.fb[r]||n.xf(t,r,1)&&(t.fb[r]=true,S[r]=true,++A));for(i=0;i<t.R.g;i++)t.V[i]&&(t.p[i]||n.sf(t,i,1)&&(t.p[i]=true,p[i]=true,++A));if(A==1){for(e=0;e<t.R.f;e++)if(S[e]){t.db[e]=0;break;}for(P=0;P<t.R.g;P++)if(p[P]){t.o[P]=0;break;}}else if(A>1)for(n.Bf(t),t.T=0,u=new U.rrb(t.A);u.a<u.c.a.length;){for(D=U.qrb(u),E=0,V=0,L=0,l=0,y=-1,Q=-1,b=0;b<D.a.length;b++)S[D.a[b]]&&(++E,(t.db[D.a[b]]==1||t.db[D.a[b]]==2)&&(++V,q=true,y<t.f[D.a[b]]&&(y=t.f[D.a[b]],L=D.a[b])));for(g=0;g<D.b.length;g++)p[D.b[g]]&&(++E,G=t.f[n.Zl(t.R,0,D.b[g])],K=t.f[n.Zl(t.R,1,D.b[g])],T=G>K?(G<<16)+K:(K<<16)+G,(t.o[D.b[g]]==1||t.o[D.b[g]]==2)&&(++V,q=true,Q<T&&(Q=T,l=D.b[g])));if(E!=0)if(E==1){for(k=0;k<D.a.length;k++)S[D.a[k]]&&(t.db[D.a[k]]=0);for(c=0;c<D.b.length;c++)p[D.b[c]]&&(t.o[D.b[c]]=0);}else if(V==1){for(k=0;k<D.a.length;k++)S[D.a[k]]&&(t.db[D.a[k]]=3);for(c=0;c<D.b.length;c++)p[D.b[c]]&&(t.o[D.b[c]]=3);}else {if((t.Q&128)!=0){for(++t.T,k=0;k<D.a.length;k++)S[D.a[k]]&&(t.Y[D.a[k]]=t.T);for(c=0;c<D.b.length;c++)p[D.b[c]]&&(t.X[D.b[c]]=t.T);}if(v=false,y!=-1?t.db[L]==2&&(v=true):t.o[l]==2&&(v=true),v){for(k=0;k<D.a.length;k++)if(S[D.a[k]])switch(t.db[D.a[k]]){case 1:t.db[D.a[k]]=2;break;case 2:t.db[D.a[k]]=1;}for(c=0;c<D.b.length;c++)if(p[D.b[c]])switch(t.o[D.b[c]]){case 1:t.o[D.b[c]]=2;break;case 2:t.o[D.b[c]]=1;}}}}return q;},n.Ff=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;for(i=U1(o.Q7,[E1,H],[7,6],15,[2,32],2),D=0;D<2;D++){for(e=B(o.Q7,E1,7,32,0,2),T=0,P=0;P<32;P++)if(A[D][P]!=null){for(L=A[D][P].length,e[P]=B(o.Q7,H,6,L,15,1),u=0;u<L;u++)e[P][u]=t.f[A[D][P][u]];w.lAb(e[P],w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),++T;}for(y=T;y>0;y--){for(l=0,Q=null,r=0;r<32;r++)if(e[r]!=null){if(Q==null||Q.length<e[r].length)Q=e[r],l=r;else if(Q.length==e[r].length){for(u=Q.length-1;u>=0;u--)if(Q[u]<e[r][u]){Q=e[r],l=r;break;}}}i[D][l]=y,e[l]=null;}}return i;},n.Gf=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p;if(y=false,t.R.K){for(l=0;l<t.R.g;l++)if(n.hm(t.R,l)!=0){y=true;break;}}for(t.O=2,e=0;e<t.R.f;e++)t.O=d.Math.max(t.O,n.$o(t.R,e)+n.rp(t.R,e));for(Q=d.Math.max(2,y?(62+t.c+t.O*(t.c+23))/63|0:(62+t.c+t.O*(t.c+5))/63|0),t.f=B(o.Q7,H,6,t.R.q,15,1),t.e=B(o.a8,Hn,101,t.R.f,0,1),r=0;r<t.R.f;r++)t.e[r]=new n.fh(Q);for(u=false,P=0;P<t.R.f;P++)n.eh(t.e[P],P),pt(J(n.Ol(t.R,P),1),0)||n.Jl(t.R,P)!=null?n.ah(t.e[P],8,6):n.ah(t.e[P],8,Ct(n.Tl(t.R,P))),n.ah(t.e[P],8,Ct(n.Ml(t.R,P))),n.ah(t.e[P],2,Ct(n.Qo(t.R,P))),n.ah(t.e[P],4,Ct(n.$o(t.R,P)+n.rp(t.R,P))),pt(J(n.Ol(t.R,P),1),0)?n.ah(t.e[P],4,8):n.ah(t.e[P],4,Ct(8+n.Bl(t.R,P))),n.ah(t.e[P],5,Ct(d.Math.min(31,n.Uo(t.R,P)))),n.ah(t.e[P],4,Ct(n.vf(t,P)+1)),n.ah(t.e[P],2,Ct(n.Pl(t.R,P)>>4)),t.R.K&&(n.ah(t.e[P],51,n.Ol(t.R,P)),n.Jl(t.R,P)!=null&&(u=true));if(t.U=n.Kf(t),t.U<t.R.f){for(A=0;A<t.R.f;A++){for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),T=B(o.Q7,H,6,n.$o(t.R,A),15,1),b=0;b<n.$o(t.R,A);b++)T[b]=t.f[n.Zo(t.R,A,b)]<<5,T[b]|=d.Math.min(31,n.Yo(t.R,n._o(t.R,A,b)));for(w.lAb(T,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),g=t.O;g>T.length;g--)n.ah(t.e[A],t.c+5,0);for(c=T.length-1;c>=0;c--)n.ah(t.e[A],t.c+5,o.Kfb(T[c]));}t.U=n.Kf(t);}if(u&&t.U<t.R.f){for(A=0;A<t.R.f;A++){for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),D=n.Jl(t.R,A),S=D==null?0:d.Math.min(12,D.length),b=12;b>S;b--)n.ah(t.e[A],8,0);for(c=S-1;c>=0;c--)n.ah(t.e[A],8,o.Kfb(D[c]));}t.U=n.Kf(t);}if(y&&t.U<t.R.f){for(A=0;A<t.R.f;A++){for(n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),L=B(o.R7,e0,6,n.$o(t.R,A)+n.rp(t.R,A),14,1),k=0,b=0;b<n.Mo(t.R,A);b++)(b<n.$o(t.R,A)||b>=n.Lo(t.R,A))&&(L[k]=Ct(t.f[n.Zo(t.R,A,b)]),L[k]=h1(L[k],23),L[k]=Pt(L[k],Ct(n.hm(t.R,n._o(t.R,A,b)))),++k);for(w.lAb(L,w.SAb(T0(U.$rb.prototype.md,U.$rb,[]))),g=t.O;g>L.length;g--)n.ah(t.e[A],t.c+23,0);for(c=L.length-1;c>=0;c--)n.ah(t.e[A],t.c+23,L[c]);}t.U=n.Kf(t);}if((t.Q&8)!=0&&t.U<t.R.f){for(v=new n.Zv(),i=0;i<t.R.f;i++)n.El(t.R,i)!=null&&n.Xv(v,n.El(t.R,i));for(A=0;A<t.R.f;A++)p=n.El(t.R,A)==null?0:1+n.Yv(v,n.El(t.R,A)),n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),n.ah(t.e[A],t.c,o.Kfb(p));t.U=n.Kf(t);}if((t.Q&16)!=0&&t.U<t.R.f){for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),n.ah(t.e[A],1,Ct(n.Pm(t.R,A)?1:0));t.U=n.Kf(t);}(t.Q&512)!=0&&t.R.K&&n.jf(t);},n.Hf=function(t){var A,e,r,P,i,D,u,Q,l;for(e=0;e<t.R.f;e++)n.eh(t.e[e],e),n.ah(t.e[e],2*t.c+4,h1(Ct(t.f[e]),t.c+4));for(Q=1;Q<=t.U;Q++){for(P=false,r=0;r<t.R.f;r++)t.f[r]==Q&&(P=P|n.xf(t,r,2));if(P){if(u=t.U,t.U=n.Kf(t),t.U!=u)return  true;for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],2*t.c+4,h1(Ct(t.f[A]),t.c+4));}}for(l=B(o.b8,Hn,173,t.R.g,0,1),D=0;D<l.length;D++)l[D]=new n.ih(t.f[n.Zl(t.R,0,D)],t.f[n.Zl(t.R,1,D)],D);for(U.Rrb(l,0,l.length,null),i=0;i<l.length;i++)if(n.sf(t,l[i].a,2)){for(;i+1<l.length&&n.hh(l[i],l[i+1])==0;)n.sf(t,l[++i].a,2);if(u=t.U,t.U=n.Kf(t),t.U!=u)return  true;for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],2*t.c+4,h1(Ct(t.f[A]),t.c+4));}return  false;},n.If=function(t){var A,e,r,P,i,D,u,Q,l,L;for(i=0,e=0;e<t.R.f;e++)t.bb[e]!=0&&(t.bb[e]!=2||(t.Q&256)==0)&&++i;if(i!=0){for(L=B(o.Q7,H,6,i,15,1),i=0,r=0;r<t.R.f;r++)t.bb[r]!=0&&(t.bb[r]!=2||(t.Q&256)==0)&&(L[i]=t.bb[r]<<29|t.ab[r]<<24|t.f[r]<<12|r,++i);for(w.lAb(L,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),D=0,l=0,u=L[0]&G0;;)if(++l,l==L.length||u!=(L[l]&G0)){for(P=B(o.Q7,H,6,l-D,15,1),Q=D;Q<l;Q++)A=L[Q]&4095,P[Q-D]=A,t.gb[A]=true;if(U.sj(t.hb,P),l==L.length)break;u=L[l]&G0,D=l;}}},n.Jf=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c;for(u=false,Q=0;Q<t.hb.a.length;Q++){for(D=U.xj(t.hb,Q),A=true,c=-1,l=false,y=0;y<D.length;y++){if(e=D[y],t.db[e]==0){A=false;break;}if(t.db[e]!=3){for(L=true,T=0;T<D.length;T++)if(T!=y&&t.f[e]==t.f[D[T]]){L=false;break;}L&&c<t.f[e]&&(c=t.f[e],l=t.db[e]==1);}}if(A&&c!=-1){for(r=D,P=0,i=r.length;P<i;++P)e=r[P],(t.db[e]==1||t.db[e]==2)&&(t.eb[e]=l),t.gb[e]=false;U.Fj(t.hb,D),u=true,--Q;}}return u;},n.Kf=function(t){var A,e;A=n.Af(t);do e=A,n.wf(t),A=n.Af(t);while(e!=A);return A;},n.Lf=function(t){for((t.Q&1)!=0&&(t.Q&2)==0&&(t.g=U.zrb(t.f,t.R.f)),(t.Q&B1)==0&&(t.W=B(o.vfb,Tt,6,t.R.f,16,1),t.V=B(o.vfb,Tt,6,t.R.g,16,1),t.U<t.R.f&&(n.kf(t),(t.Q&B1)==0&&(n.Jf(t),t.P&&n.uh(t.P,t.f)))),t.g==null&&(t.Q&1)!=0&&(t.Q&2)!=0&&(t.g=U.zrb(t.f,t.R.f));t.U<t.R.f;)n.lf(t),(t.Q&B1)==0&&(n.Jf(t),t.P&&n.uh(t.P,t.f));(t.Q&B1)==0&&(n.Jf(t),n.Ef(t),n.hg(t));},n.Mf=function(t){var A,e,r,P,i;for(i=t.U,P=U.zrb(t.f,t.R.f),t.R.K||(n.Of(t),n.Cg(t,i,P)),t.bb=B(o.M7,Y,6,t.R.f,15,1),t.ab=B(o.M7,Y,6,t.R.f,15,1),e=0;e<t.R.f;e++)t.bb[e]=n.Hl(t.R,e)<<24>>24,t.ab[e]=n.Gl(t.R,e)<<24>>24;for(t.n=B(o.M7,Y,6,t.R.g,15,1),t.k=B(o.M7,Y,6,t.R.g,15,1),r=0;r<t.R.g;r++)t.n[r]=n.cm(t.R,r)<<24>>24,t.k[r]=n.bm(t.R,r)<<24>>24;for(n.Nf(t),t.Z=false,t.M=B(o.vfb,Tt,6,t.R.f,16,1),A=0;A<t.R.f;A++)t.db[A]!=0&&(t.M[A]=true,t.Z=true);n.Qf(t),t.P=null,t.cb=B(o.vfb,Tt,6,t.R.f,16,1),t.Z&&(t.P=new n.Ah(t.R,P,t.M,t.db,t.o,t.bb,t.ab,t.ib,t.q,t.cb),n.vh(t.P)),t.eb=B(o.vfb,Tt,6,t.R.f,16,1),t.gb=B(o.vfb,Tt,6,t.R.f,16,1),t.hb=new U.Mj(),t.P&&n.uh(t.P,t.f),n.If(t),n.Cg(t,i,P),n.Pf(t),t.P&&(t.K=n.rh(t.P)),n.$f(t);},n.Nf=function(t){var A,e,r,P,i,D;for(t.L=true,i=n.Df(t,false);t.U<t.R.f&&i;){for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),D=t.db[A]<<7,(t.db[A]==1||t.db[A]==2)&&t.bb[A]!=0&&(D|=t.bb[A]<<5,D|=t.ab[A]),n.ah(t.e[A],18,h1(o.Kfb(D),9));for(e=0;e<t.R.g;e++)r=t.o[e]<<7,(t.o[e]==1||t.o[e]==2)&&n.im(t.R,e)==1&&t.n[e]!=0&&(r|=t.n[e]<<5,r|=t.k[e]),n.bh(t.e[n.Zl(t.R,0,e)],o.Kfb(r)),n.bh(t.e[n.Zl(t.R,1,e)],o.Kfb(r));if(P=n.Kf(t),t.U==P)break;t.U=P,i=n.Df(t,false);}},n.Of=function(t){var A,e,r,P;for(t.L=true,t.$=B(o.M7,Y,6,t.R.f,15,1),t.i=B(o.M7,Y,6,t.R.g,15,1),P=n.Df(t,true);t.U<t.R.f&&P;){for(A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c+4,Pt(h1(Ct(t.f[A]),4),Ct(t.db[A]<<2)));for(e=0;e<t.R.g;e++)n.bh(t.e[n.Zl(t.R,0,e)],Ct(t.o[e])),n.bh(t.e[n.Zl(t.R,1,e)],Ct(t.o[e]));if(r=n.Kf(t),t.U==r)break;t.U=r,P=n.Df(t,true);}},n.Pf=function(t){var A,e,r,P,i,D,u;for(t.L=true,r=n.Zf(t),t.P&&n.uh(t.P,t.f)&&(r=n.Zf(t)),n.Df(t,false)&&n.Jf(t),D=true;t.U<t.R.f&&D;){for(P=n.Ff(t,r),A=0;A<t.R.f;A++)n.eh(t.e[A],A),n.ah(t.e[A],t.c,Ct(t.f[A])),n.ah(t.e[A],20,0),!t.cb[A]&&t.bb[A]!=0&&n.bh(t.e[A],v1(Ct(t.bb[A]<<18),h1(Ct(P[t.bb[A]==1?0:1][t.ab[A]]),8))),u=t.db[A],t.eb[A]&&(u==1?u=2:u==2&&(u=1)),n.bh(t.e[A],o.Kfb(u<<4));for(e=0;e<t.R.g;e++)n.bh(t.e[n.Zl(t.R,0,e)],Ct(t.o[e])),n.bh(t.e[n.Zl(t.R,1,e)],Ct(t.o[e]));if(i=n.Kf(t),t.U==i)break;t.U=i,D=false,t.P&&n.uh(t.P,t.f)&&(D=true,r=n.Zf(t)),n.Df(t,false)&&(D=true,n.Jf(t));}},n.Qf=function(t){var A,e;for(A=0;A<t.R.f;A++)(!t.M[A]||t.db[A]==3)&&(t.bb[A]=0);for(e=0;e<t.R.g;e++)(n.im(t.R,e)!=1||t.o[e]==0||t.o[e]==3)&&(t.n[e]=0);},n.Rf=function(t,A,e){t.a==null&&(t.a=B(o.M7,Y,6,t.R.f,15,1),U.Crb(t.a)),t.a[A]=e<<24>>24;},n.Sf=function(t,A){var e,r,P,i,D,u;if((t.o[A]==1||t.o[A]==2)&&!n.jq(t.R,A)){D=false;try{for(i=0;i<2;i++)if(e=n.Zl(t.R,i,A),n.$o(t.R,e)==3){for(r=B(o.Q7,H,6,2,15,1),P=0,u=0;u<n.$o(t.R,e);u++)n._o(t.R,e,u)!=A&&(r[P++]=n.Zo(t.R,e,u));t.f[r[0]]>t.f[r[1]]^n.Uf(t,e,r[0],r[1])&&(D=!D);}}catch(Q){if(Q=C1(Q),Zt(Q,19)){t.i[A]=3;return;}else throw st(Q);}t.o[A]==1^D?t.i[A]=1:t.i[A]=2;}},n.Tf=function(t,A){var e,r,P,i,D,u,Q;if(t.db[A]==1||t.db[A]==2){if(u=false,n.Qo(t.R,A)==2&&n.$o(t.R,A)==2)try{for(D=0;D<2;D++)if(e=n.Zo(t.R,A,D),n.$o(t.R,e)==3){for(P=B(o.Q7,H,6,2,15,1),i=0,Q=0;Q<n.$o(t.R,e);Q++)n.ap(t.R,e,Q)==1&&(P[i++]=n.Zo(t.R,e,Q));t.f[P[0]]>t.f[P[1]]^n.Uf(t,e,P[0],P[1])&&(u=!u);}}catch(l){if(l=C1(l),Zt(l,19)){t.$[A]=3;return;}else throw st(l);}else {try{r=n.Wf(t,A);}catch(l){if(l=C1(l),Zt(l,19)){t.$[A]=3;return;}else throw st(l);}for(D=1;D<r.length;D++)for(Q=0;Q<D;Q++)t.f[r[D]]<t.f[r[Q]]&&(u=!u);}t.db[A]==1^u?t.$[A]=1:t.$[A]=2;}},n.Uf=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot;if(n.Tl(t.R,e)!=n.Tl(t.R,r))return n.Tl(t.R,e)>n.Tl(t.R,r);if(n.Ml(t.R,e)!=n.Ml(t.R,r))return it=n.Mm(t.R,e)?(n.Rk(),n.Pk)[n.Tl(t.R,e)]:n.Ml(t.R,e),lt=n.Mm(t.R,r)?(n.Rk(),n.Pk)[n.Tl(t.R,r)]:n.Ml(t.R,r),it>lt;for(V=t.R.f,S=B(o.Q7,H,6,V,15,1),E=B(o.Q7,H,6,V,15,1),q=B(o.Q7,H,6,V,15,1),p=B(o.vfb,Tt,6,V,16,1),Q=B(o.vfb,Tt,6,t.R.q,16,1),S[0]=A,S[1]=e,S[2]=r,E[0]=-1,E[1]=0,E[2]=0,Q[A]=true,Q[e]=true,Q[r]=true,T=1,G=2,z=B(o.Q7,H,6,64,15,1),z[1]=1,z[2]=3,b=2;T<=G;){for(;T<z[b];){if(c=S[T],!p[T]){for(g=0,k=0,O=0;O<n.$o(t.R,c);O++){if(L=n.Zo(t.R,c,O),G+n.ap(t.R,c,O)+1>=V&&(V+=t.R.f,S=n.Gg(S,V),E=n.Gg(E,V),q=n.Gg(q,V),p=n.Hg(p,V)),n.Yp(t.R,n._o(t.R,c,O)))++g,k+=n.Tl(t.R,L);else if(L!=A)for(Qt=1;Qt<n.ap(t.R,c,O);Qt++)++G,S[G]=L,E[G]=T,p[G]=true;if(Dt=E[T],L!=S[Dt]){if(u=false,Q[L])for(ut=E[Dt];ut!=-1;){if(L==S[ut]){u=true;break;}ut=E[ut];}u?(++G,S[G]=L,E[G]=T,p[G]=true):(++G,S[G]=L,E[G]=T,Q[L]=true);}}g!=0&&(++G,q[G]=(k<<2)/g|0,E[G]=T,p[G]=true);}if(++T,T==Z0)throw st(new h.cC("Emergency break in while loop."));}for(z.length==b+1&&(z=n.Gg(z,z.length+64)),z[b+1]=G+1,K=z[b];K<z[b+1];K++)q[K]==0&&(q[K]=(n.Tl(t.R,S[K])==151||n.Tl(t.R,S[K])==152?1:n.Tl(t.R,S[K]))<<2),q[K]+=q[E[K]]<<16;if(n.Yf(t,p,q,E,S,z,b),q[1]!=q[2])return q[1]>q[2];b>1&&n.Vf(q,E,z,b),++b;}for(y=B(o.Q7,H,6,t.R.f,15,1),tt=false,i=0;i<t.R.f;i++)if(Q[i]&&!n.Mm(t.R,i)){tt=true;break;}if(tt){for(D=0;D<t.R.f;D++)y[D]=n.Mm(t.R,D)?(n.Rk(),n.Pk)[n.Tl(t.R,D)]:n.Ml(t.R,D);if(n.Xf(t,p,q,E,S,y,z,b))return q[1]>q[2];}for(U.Jrb(y,y.length,0),v=false,l=0;l<t.R.g;l++)(Q[n.Zl(t.R,0,l)]||Q[n.Zl(t.R,1,l)])&&(t.i[l]==1?(y[n.Zl(t.R,0,l)]=1,y[n.Zl(t.R,1,l)]=1,v=true):t.i[l]==2&&(y[n.Zl(t.R,0,l)]=2,y[n.Zl(t.R,1,l)]=2,v=true));if(v&&n.Xf(t,p,q,E,S,y,z,b))return q[1]>q[2];for(U.Jrb(y,y.length,0),ot=false,P=0;P<t.R.f;P++)Q[P]&&(t.$[P]==2?(y[P]=1,ot=true):t.$[P]==1&&(y[P]=2,ot=true));if(ot&&n.Xf(t,p,q,E,S,y,z,b))return q[1]>q[2];throw t.d=true,st(new h.cC("no distinction applying CIP rules"));},n.Vf=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T;for(y=e[r],D=e[r+1]-y,T=B(o.$7,R1,175,D,0,1),Q=0;Q<D;Q++)T[Q]=new n._g(),T[Q].c=t[Q+y],T[Q].b=A[Q+y],T[Q].a=Q+y;for(P=new n.Yg(),L=r;L>1;L--){for(l=0;l<D;l++)T[l].c+=t[T[l].b]<<16,T[l].b=A[T[l].b];for(U.Rrb(T,0,T.length,P),i=1,u=0;u<D;u++)t[T[u].a]=i,u!=D-1&&n.Xg(T[u],T[u+1])!=0&&++i;}},n.Wf=function(t,A){var e,r,P,i,D,u,Q;for(D=n.Lo(t.R,A),u=B(o.Q7,H,6,D,15,1),P=0;P<D;P++)u[P]=n.Zo(t.R,A,P);for(r=D;r>1;r--){for(e=false,i=1;i<r;i++)n.Uf(t,A,u[i-1],u[i])&&(e=true,Q=u[i-1],u[i-1]=u[i],u[i]=Q);if(!e)break;}return u;},n.Xf=function(t,A,e,r,P,i,D,u){var Q,l;for(l=1;l<u;l++){for(Q=D[l];Q<D[l+1];Q++)e[Q]=i[P[Q]]+(e[r[Q]]<<8);if(n.Yf(t,A,e,r,P,D,l),e[1]!=e[2])return  true;l>1&&n.Vf(e,r,D,l);}return  false;},n.Yf=function(t,A,e,r,P,i,D){var u,Q,l,L,y,T,c,b,g,k,v;for(y=D;y>1;y--){for(g=i[y]-i[y-1],v=B(o.Y7,R1,174,g,0,1),u=i[y],b=0;b<g;b++){for(k=i[y-1]+b,T=u;T<i[y+1]&&r[T]==k;)++T;for(v[b]=new n.Vg(),v[b].c=k,v[b].d=e[k],v[b].b=A[k]?0:n.Cp(t.R,P[k]),v[b].a=B(o.Q7,H,6,T-u,15,1),L=u;L<T;L++)v[b].a[L-u]=e[L];w.lAb(v[b].a,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),u=T;}for(Q=new n.Sg(),U.Rrb(v,0,v.length,Q),l=1,c=0;c<g;c++)e[v[c].c]=l,c!=g-1&&n.Rg(v[c],v[c+1])!=0&&++l;}},n.Zf=function(t){var A,e;for(e=U1(o.Q7,[b0,E1],[28,7],0,[2,32],2),A=0;A<t.R.f;A++)t.M[A]&&(t.bb[A]==1?e[0][t.ab[A]]=n.Bh(e[0][t.ab[A]],A):t.bb[A]==2&&(e[1][t.ab[A]]=n.Bh(e[0][t.ab[A]],A)));return e;},n.$f=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T;for(i=0,L=0,D=0,u=0,Q=0,l=0,y=0,T=false,A=B(o.vfb,Tt,6,32,16,1),e=0;e<t.R.f;e++)t.db[e]!=0&&(++i,t.db[e]==3?++L:t.bb[e]==0?(++D,t.P&&n.qh(t.P,e)&&++u):t.bb[e]==2?t.ab[e]==0&&++l:t.bb[e]==1&&(P=t.ab[e],A[P]||(++y,A[P]=true),t.ab[e]==0&&++Q,t.P&&n.qh(t.P,e)&&(T=true)));for(r=0;r<t.R.g;r++)t.o[r]!=0&&n.im(t.R,r)==1&&(++i,t.o[r]==3?++L:t.n[r]==0?(++D,t.P&&n.qh(t.P,n.Zl(t.R,0,r))&&n.qh(t.P,n.Zl(t.R,1,r))&&++u):t.n[r]==2?t.k[r]==0&&++l:t.n[r]==1&&(P=t.k[r],A[P]||(++y,A[P]=true),t.k[r]==0&&++Q,t.P&&n.qh(t.P,n.Zl(t.R,0,r))&&n.qh(t.P,n.Zl(t.R,1,r))&&(T=true)));if(i==0){n.Nn(t.R,H1);return;}if(L!=0){n.Nn(t.R,0);return;}if(t.K){n.Nn(t.R,r0+(1<<y));return;}Q+u==i&&!T?n.Nn(t.R,Ye):D==i?n.Nn(t.R,yA):l==i?n.Nn(t.R,327680):D==i-1&&Q==1?n.Nn(t.R,tn):n.Nn(t.R,458752+(1<<y));},n._f=function(t,A,e){for(;e!=0;)t.t==0&&((!t.r||t.v!=63)&&(t.v+=64),h.Ekb(t.u,t.v&e1),t.t=6,t.v=0),t.v<<=1,t.v=Jt(Pt(Ct(t.v),J(A,1))),A=A0(A,1),--e,--t.t;},n.ag=function(t){return t.v<<=t.t,(!t.r||t.v!=63)&&(t.v+=64),h.Ekb(t.u,t.v&e1),t.u.a;},n.bg=function(t,A){t.u=new h.Kkb(),t.t=6,t.v=0,t.r=A;},n.cg=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k;if(t.R.f==0){t.s="";return;}for(y=t.jb&t.b,k=t.jb?16:8,n.bg(t,true),h.Ekb(t.u,y?35:33),n._f(t,Ct(t.jb?1:0),1),n._f(t,o.Kfb(A?1:0),1),n._f(t,o.Kfb(k>>1),4),b=0,Q=1;Q<t.R.f;Q++)b=n.sg(t,t.B[Q],t.F[Q]==-1?-1:t.B[t.F[Q]],b,e);if(y)for(l=0;l<t.R.f;l++)for(r=t.B[l],c=n.$o(t.R,r);c<n.Lo(t.R,r);c++)b=n.sg(t,n.Zo(t.R,r,c),r,b,e);if(t.R.f>1&&b==0){t.s="";return;}for(D=1<<k,T=b/((D>>1)-1),g=b+.5*T,L=1;L<t.R.f;L++)n.dg(t,t.B[L],t.F[L]==-1?-1:t.B[t.F[L]],g,T,k,e);if(y)for(u=0;u<t.R.f;u++)for(r=t.B[u],c=n.$o(t.R,r);c<n.Lo(t.R,r);c++)n.dg(t,n.Zo(t.R,r,c),r,g,T,k,e);A&&(i=t.jb?1.5:(n.Rk(),n.Rk(),n.Qk),P=n.Xl(t.R,y?t.R.q:t.R.f,y?t.R.r:t.R.g,i,e),n._f(t,Ct(d.Math.min(D-1,d.Math.max(0,Mt(.5+d.Math.log(P/.1)*d.Math.LOG10E/(d.Math.log(2e3)*d.Math.LOG10E)*(D-1))))),k),n._f(t,Ct(n.gg(e[t.B[0]].a/P,D)),k),n._f(t,Ct(n.gg(e[t.B[0]].b/P,D)),k),t.jb&&n._f(t,Ct(n.gg(e[t.B[0]].c/P,D)),k)),t.s=n.ag(t);},n.dg=function(t,A,e,r,P,i,D){var u,Q,l;u=e==-1?(D[A].a-D[t.B[0]].a)/8:D[A].a-D[e].a,Q=e==-1?(D[A].b-D[t.B[0]].b)/8:D[A].b-D[e].b,n._f(t,Ct(Mt((r+u)/P)),i),n._f(t,Ct(Mt((r+Q)/P)),i),t.jb&&(l=e==-1?(D[A].c-D[t.B[0]].c)/8:D[A].c-D[e].c,n._f(t,Ct(Mt((r+l)/P)),i));},n.eg=function(t,A){var e;for(e=0;e<t.w;e++)A-=16;for(A<0&&h.Pkb();A>15;)n._f(t,1,1),n._f(t,15,4),A-=16,++t.w;n._f(t,1,1),n._f(t,o.Kfb(A),4);},n.fg=function(t){var A,e,r,P,i,D;if(t.R.f==0){t.N="";return;}for(i=0,r=false,P=false,e=0;e<t.R.f;e++)i<n.Ll(t.R,e)&&(i=n.Ll(t.R,e)),n.Am(t.R,e)?r=true:P=true;if(i==0){t.N="";return;}for(D=n.Pg(i),n.bg(t,true),n._f(t,o.Kfb(D),4),n._f(t,o.Kfb(r?1:0),1),n._f(t,o.Kfb(P?1:0),1),A=0;A<t.R.f;A++)n._f(t,Ct(n.Ll(t.R,t.B[A])),D),r&&P&&n._f(t,Ct(n.Am(t.R,t.B[A])?1:0),1);t.N=n.ag(t);},n.gg=function(t,A){var e,r,P,i;return e=A/2|0,P=t<0,t=d.Math.abs(t),i=A/32|0,r=d.Math.min(e-1,Jt(b1(d.Math.round(t*e/(t+i))))),P?e+r:r;},n.hg=function(t){var A,e,r;for(A=0;A<t.R.f;A++)t.db[A]==3&&!n.wm(t.R,A)&&n.Vn(t.R,A),(n.Hl(t.R,A)==1||n.Hl(t.R,A)==2)&&t.db[A]==3&&n.Vn(t.R,A),n.wm(t.R,A)&&t.db[A]!=3&&!n.Eg(t,A)&&n.Vn(t.R,A);for(r=0;r<t.R.r;r++)n.Rm(t.R,r)&&!n.Dg(t,r)&&n.Vn(t.R,n.Zl(t.R,0,r));for(e=0;e<t.R.g;e++)n.fm(t.R,e)==2&&(n.Gm(t.R,e)&&(t.o[e]==1||t.o[e]==2)&&(t.o[e]=3,n.Mn(t.R,e,386)),t.o[e]==3&&!t.p[e]&&n.im(t.R,e)!=386&&(n.Vn(t.R,n.Zl(t.R,0,e)),n.Vn(t.R,n.Zl(t.R,1,e)))),n.im(t.R,e)==1&&t.o[e]==3&&!n.wm(t.R,n.Zl(t.R,0,e))&&!n.wm(t.R,n.Zl(t.R,1,e))&&(n.Vn(t.R,n.Zl(t.R,0,e)),n.Vn(t.R,n.Zl(t.R,1,e))),(n.cm(t.R,e)==1||n.cm(t.R,e)==2)&&(n.im(t.R,e)!=1||t.o[e]!=1&&t.o[e]!=2)&&(n.Vn(t.R,n.Zl(t.R,0,e)),n.Vn(t.R,n.Zl(t.R,1,e)));},n.ig=function(t){var A,e,r,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;if(t.R.f!=0&&!t.G){for(t.I=0,q=0,e=1;e<t.R.f;e++)t.f[e]>t.f[q]&&(q=e);for(r=B(o.vfb,Tt,6,t.R.f,16,1),D=B(o.vfb,Tt,6,t.R.g,16,1),t.H=B(o.Q7,H,6,t.R.f,15,1),t.B=B(o.Q7,H,6,t.R.f,15,1),t.F=B(o.Q7,H,6,t.R.f,15,1),t.C=B(o.Q7,H,6,t.R.g,15,1),t.B[0]=q,t.H[q]=0,r[q]=true,Q=0,l=1,L=0;Q<t.R.f;)if(Q<l){for(;;){for(b=0,g=0,T=-1,A=t.B[Q],k=0;k<n.Mo(t.R,A);k++)(k<n.$o(t.R,A)||k>=n.Lo(t.R,A))&&(u=n.Zo(t.R,A,k),!r[u]&&t.f[u]>T&&(b=u,g=n._o(t.R,A,k),T=t.f[u]));if(T==-1)break;t.H[b]=l,t.F[l]=Q,t.B[l++]=b,t.C[L++]=g,r[b]=true,D[g]=true;}++Q;}else {for(c=0,T=-1,A=0;A<t.R.f;A++)!r[A]&&t.f[A]>T&&(c=A,T=t.f[A]);t.H[c]=l,t.F[l]=-1,t.B[l++]=c,r[c]=true;}for(t.D=B(o.Q7,H,6,2*(t.R.g-L),15,1);;){for(S=t.R.M,p=t.R.M,E=-1,i=0;i<t.R.g;i++)D[i]||(t.H[n.Zl(t.R,0,i)]<t.H[n.Zl(t.R,1,i)]?(v=t.H[n.Zl(t.R,0,i)],y=t.H[n.Zl(t.R,1,i)]):(v=t.H[n.Zl(t.R,1,i)],y=t.H[n.Zl(t.R,0,i)]),(v<S||v==S&&y<p)&&(S=v,p=y,E=i));if(E==-1)break;D[E]=true,t.C[L++]=E,t.D[2*t.I]=S,t.D[2*t.I+1]=p,++t.I;}t.G=true;}},n.jg=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b;for(Q=null,b=n.Ep(t.R),y=0;y<b.j.a.length;y++)if(b.f[y]){for(i=0,T=U.xj(b.j,y),e=T,r=0,P=e.length;r<P;++r)A=e[r],n.yg(t,A)&&++i;if(i!=0)if(c=U.xj(b.k,y),Q==null&&(Q=B(o.vfb,Tt,6,t.R.g,16,1)),i==T.length){for(l=-1,L=qt,D=0;D<T.length;D++)L>t.B[c[D]]&&(L=t.B[c[D]],l=D);for(;i>0;)Q[c[l]]=true,l=n.Mg(l+2,T.length),i-=2;}else {for(u=0;n.yg(t,T[u]);)++u;for(;!n.yg(t,T[u]);)u=n.Mg(u+1,T.length);for(;i>0;)for(Q[c[u]]=true,u=n.Mg(u+2,T.length),i-=2;!n.yg(t,T[u]);)u=n.Mg(u+1,T.length);}}return Q;},n.kg=function(t,A){var e,r,P,i,D,u,Q,l,L;for(n.ig(t),l=new n.uw(t.R.f,t.R.g),n.On(l,t.R.K),D=0;D<t.R.f;D++)n.ll(t.R,l,t.B[D],0,0),n.on(l,D,t.bb[t.B[D]],t.ab[t.B[D]]);for(u=0;u<t.R.g;u++)n.nl(t.R,l,t.C[u],0,0,t.H,false),!(l.H[u]==257||l.H[u]==129)&&l.D[0][u]>l.D[1][u]&&(L=l.D[0][u],n.Dn(l,0,u,l.D[1][u]),l.D[1][u]=L,l.T=0),n.Gn(l,u,t.n[t.C[u]],t.k[t.C[u]]);if(A)for(i=0;i<t.R.f;i++)for(e=t.B[i],Q=n.$o(t.R,e);Q<n.Lo(t.R,e);Q++)P=n.ll(t.R,l,n.Zo(t.R,e,Q),0,0),n.ml(t.R,l,n._o(t.R,e,Q),0,0,t.H[e],P,false);for(r=0;r<l.r;r++)e=l.D[0][r],t.eb[t.B[e]]&&(l.H[r]==257?(l.H[r]=129,l.T=0):l.H[r]==129&&(l.H[r]=257,l.T=0));return n.aw(t.R,l),l.T&=-9,l;},n.lg=function(t,A){return t.o[A];},n.mg=function(t){return n.ng(t,t.jb);},n.ng=function(t,A){return t.s==null&&(n.ig(t),n.cg(t,A,t.R.J)),t.s;},n.og=function(t){return t.N==null&&(n.ig(t),n.fg(t)),t.N;},n.pg=function(t){return n.ig(t),t.B;},n.qg=function(t){return n.ig(t),t.H;},n.rg=function(t){return t.J==null&&(n.ig(t),(t.Q&B1)==0&&(n.Ag(t),n.Bg(t,1),n.Bg(t,2)),n.zg(t)),t.J;},n.sg=function(t,A,e,r,P){var i,D,u;return i=e==-1?d.Math.abs(P[A].a-P[t.B[0]].a)/8:d.Math.abs(P[A].a-P[e].a),r<i&&(r=i),D=e==-1?d.Math.abs(P[A].b-P[t.B[0]].b)/8:d.Math.abs(P[A].b-P[e].b),r<D&&(r=D),t.jb&&(u=e==-1?d.Math.abs(P[A].c-P[t.B[0]].c)/8:d.Math.abs(P[A].c-P[e].c),r<u&&(r=u)),r;},n.tg=function(t,A){return t.X[A];},n.ug=function(t,A){return t.Y[A];},n.vg=function(t,A){return t.g==null?-1:t.g[A];},n.wg=function(t,A){return t.db[A];},n.xg=function(t,A){var e,r,P,i,D,u,Q;for(Q=n.Ep(t.R),e=0;e<Q.j.a.length;e++)if(Q.e[e]&&n.qt(Q,e,A)){for(P=U.xj(Q.j,e),i=0,D=P.length;i<D;++i)if(r=P[i],r!=A){for(u=0;u<n.$o(t.R,r);u++)if(n.Vp(t.R,n._o(t.R,r,u)))return  true;}return  false;}return  false;},n.yg=function(t,A){var e,r,P;if(n.Qo(t.R,A)<2)return  false;if(n.$o(t.R,A)==2)return  true;for(e=0,P=0;P<n.$o(t.R,A);P++)r=n._o(t.R,A,P),n.Up(t.R,r)&&(e+=n.fm(t.R,r)-1);return e>1;},n.zg=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt,c1,D1,St,gt;if(n.bg(t,false),n._f(t,9,4),Bt=d.Math.max(n.Pg(t.R.f),n.Pg(t.R.g)),n._f(t,o.Kfb(Bt),4),Bt==0){n._f(t,Ct(t.R.K?1:0),1),n._f(t,0,1),t.J=n.ag(t);return;}for(c1=St=D1=it=0,Q=0;Q<t.R.f;Q++)if(S1(J(n.Ol(t.R,Q),1),0)){switch(n.Tl(t.R,Q)){case 6:break;case 7:++c1;break;case 8:++St;break;default:++D1;}n.Bl(t.R,Q)!=0&&++it;}for(n._f(t,Ct(t.R.f),Bt),n._f(t,Ct(t.R.g),Bt),n._f(t,o.Kfb(c1),Bt),n._f(t,o.Kfb(St),Bt),n._f(t,o.Kfb(D1),Bt),n._f(t,o.Kfb(it),Bt),l=0;l<t.R.f;l++)n.Tl(t.R,t.B[l])==7&&S1(J(n.Ol(t.R,t.B[l]),1),0)&&n._f(t,o.Kfb(l),Bt);for(T=0;T<t.R.f;T++)n.Tl(t.R,t.B[T])==8&&S1(J(n.Ol(t.R,t.B[T]),1),0)&&n._f(t,o.Kfb(T),Bt);for(c=0;c<t.R.f;c++)n.Tl(t.R,t.B[c])!=6&&n.Tl(t.R,t.B[c])!=7&&n.Tl(t.R,t.B[c])!=8&&S1(J(n.Ol(t.R,t.B[c]),1),0)&&(n._f(t,o.Kfb(c),Bt),n._f(t,Ct(n.Tl(t.R,t.B[c])),8));for(b=0;b<t.R.f;b++)n.Bl(t.R,t.B[b])!=0&&S1(J(n.Ol(t.R,t.B[b]),1),0)&&(n._f(t,o.Kfb(b),Bt),n._f(t,Ct(8+n.Bl(t.R,t.B[b])),4));for(kt=0,q=0,g=1;g<t.R.f;g++)t.F[g]==-1?Ut=0:(Ut=1+t.F[g]-q,q=t.F[g]),kt<Ut&&(kt=Ut);for(bt=n.Pg(kt),n._f(t,o.Kfb(bt),4),q=0,k=1;k<t.R.f;k++)t.F[k]==-1?Ut=0:(Ut=1+t.F[k]-q,q=t.F[k]),n._f(t,o.Kfb(Ut),bt);for(It=0;It<2*t.I;It++)n._f(t,Ct(t.D[It]),Bt);for(G=0;G<t.R.g;G++)z=(n.hm(t.R,t.C[G])&sA)!=0||n.im(t.R,t.C[G])==16?1:n.Yp(t.R,t.C[G])?0:d.Math.min(3,n.fm(t.R,t.C[G])),n._f(t,o.Kfb(z),2);if(e=0,(t.Q&B1)==0)for(v=0;v<t.R.f;v++)t._[t.B[v]]!=0&&t._[t.B[v]]!=3&&++e;if(n._f(t,o.Kfb(e),Bt),(t.Q&B1)==0)for(v=0;v<t.R.f;v++)t._[t.B[v]]!=0&&t._[t.B[v]]!=3&&(n._f(t,o.Kfb(v),Bt),t.bb[t.B[v]]==0?n._f(t,Ct(t._[t.B[v]]),3):(gt=t._[t.B[v]]==1?t.bb[t.B[v]]==1?4:6:t.bb[t.B[v]]==1?5:7,n._f(t,o.Kfb(gt),3),n._f(t,Ct(t.ab[t.B[v]]),3)));if(A=0,(t.Q&B1)==0)for(K=0;K<t.R.g;K++)t.j[t.C[K]]!=0&&t.j[t.C[K]]!=3&&(!n.jq(t.R,t.C[K])||n.im(t.R,t.C[K])==1)&&++A;if(n._f(t,o.Kfb(A),Bt),(t.Q&B1)==0)for(K=0;K<t.R.g;K++)t.j[t.C[K]]!=0&&t.j[t.C[K]]!=3&&(!n.jq(t.R,t.C[K])||n.im(t.R,t.C[K])==1)&&(n._f(t,o.Kfb(K),Bt),n.im(t.R,t.C[K])==1?t.n[t.C[K]]==0?n._f(t,Ct(t.j[t.C[K]]),3):(gt=t.j[t.C[K]]==1?t.n[t.C[K]]==1?4:6:t.n[t.C[K]]==1?5:7,n._f(t,o.Kfb(gt),3),n._f(t,Ct(t.k[t.C[K]]),3)):n._f(t,Ct(t.j[t.C[K]]),2));for(n._f(t,Ct(t.R.K?1:0),1),Dt=0,S=0;S<t.R.f;S++)n.Ml(t.R,t.B[S])!=0&&++Dt;if(Dt!=0)for(n.eg(t,1),n._f(t,o.Kfb(Dt),Bt),v=0;v<t.R.f;v++)n.Ml(t.R,t.B[v])!=0&&(n._f(t,o.Kfb(v),Bt),n._f(t,Ct(n.Ml(t.R,t.B[v])),8));if(t.w=0,t.R.K){for(n.gf(t,0,Bt,B1,1,-1),n.gf(t,3,Bt,Z1,1,-1),n.gf(t,4,Bt,120,4,3),n.gf(t,5,Bt,Xe,2,1),n.gf(t,6,Bt,1,1,-1),n.gf(t,7,Bt,X0,4,7),Dt=0,v=0;v<t.R.f;v++)n.Jl(t.R,t.B[v])!=null&&++Dt;if(Dt>0){for(n.eg(t,8),n._f(t,o.Kfb(Dt),Bt),p=0;p<t.R.f;p++)if(E=n.Jl(t.R,t.B[p]),E!=null)for(n._f(t,o.Kfb(p),Bt),n._f(t,Ct(E.length),4),P=E,i=0,D=P.length;i<D;++i)r=P[i],n._f(t,o.Kfb(r),8);}n.hf(t,9,Bt,384,2,7),n.hf(t,10,Bt,31,5,0),n.gf(t,11,Bt,L0,1,-1),n.hf(t,12,Bt,sA,8,9),n.gf(t,13,Bt,$A,3,14),n.gf(t,14,Bt,aA,5,17),n.gf(t,16,Bt,vA,3,22);}for(Dt=0,L=0;L<t.R.f;L++)t.a!=null&&t.a[t.B[L]]!=-1&&++Dt;if(Dt!=0)for(n.eg(t,17),n._f(t,o.Kfb(Dt),Bt),v=0;v<t.R.f;v++)t.a!=null&&t.a[t.B[v]]!=-1&&(n._f(t,o.Kfb(v),Bt),n._f(t,Ct(t.a[t.B[v]]),4));if((t.Q&8)!=0||(t.Q&f0)!=0){for(Dt=0,Nt=0,v=0;v<t.R.f;v++)Kt=n.El(t.R,t.B[v]),Kt!=null&&(++Dt,Nt=d.Math.max(Nt,w.SAb(Kt).length));if(Dt!=0){for(xt=n.Pg(Nt),n.eg(t,18),n._f(t,o.Kfb(Dt),Bt),n._f(t,o.Kfb(xt),4),p=0;p<t.R.f;p++)if(ot=n.El(t.R,t.B[p]),ot!=null)for(n._f(t,o.Kfb(p),Bt),n._f(t,Ct(w.SAb(ot).length),xt),ct=0;ct<w.SAb(ot).length;ct++)n._f(t,(w.NAb(ct,w.SAb(ot).length),Ct(w.SAb(ot).charCodeAt(ct))),7);}}for(t.R.K&&(n.gf(t,19,Bt,mA,3,25),n.hf(t,20,Bt,iA,3,17)),Dt=0,y=0;y<t.R.f;y++)n.Pl(t.R,t.B[y])!=0&&++Dt;if(Dt!=0)for(n.eg(t,21),n._f(t,o.Kfb(Dt),Bt),u=0;u<t.R.f;u++)n.Pl(t.R,t.B[u])!=0&&(n._f(t,o.Kfb(u),Bt),n._f(t,Ct(n.Pl(t.R,t.B[u])>>4),2));if(t.R.K&&(n.gf(t,22,Bt,NA,1,-1),n.hf(t,23,Bt,TA,1,-1),n.hf(t,24,Bt,Qe,2,21)),(t.Q&16)!=0){for(u=0;u<t.R.f;u++)if(n.Pm(t.R,t.B[u])){for(n.eg(t,25),r=0;r<t.R.f;r++)n._f(t,Ct(n.Pm(t.R,t.B[r])?1:0),1);break;}}if(Et=n.jg(t),Et!=null){for(Dt=0,K=0;K<t.R.g;K++)Et[t.C[K]]&&++Dt;for(n.eg(t,26),n._f(t,o.Kfb(Dt),Bt),O=0;O<t.R.g;O++)Et[t.C[O]]&&n._f(t,o.Kfb(O),Bt);}for(t.R.K&&n.gf(t,27,Bt,i0,1,-1),Dt=0,tt=0;tt<t.R.g;tt++)n.im(t.R,t.C[tt])==16&&++Dt;if(Dt!=0)for(n.eg(t,28),n._f(t,o.Kfb(Dt),Bt),K=0;K<t.R.g;K++)n.im(t.R,t.C[K])==16&&n._f(t,o.Kfb(K),Bt);for(t.R.K&&(n.gf(t,29,Bt,hA,2,30),n.gf(t,30,Bt,EA,7,32),n.gf(t,32,Bt,DP,2,44),n.gf(t,33,Bt,Ln,5,39),n.gf(t,34,Bt,ee,1,-1),n.hf(t,35,Bt,sn,1,-1),n.hf(t,36,Bt,96,2,5)),Dt=0,Qt=0;Qt<t.R.g;Qt++)(n.im(t.R,t.C[Qt])==32||n.im(t.R,t.C[Qt])==64)&&++Dt;if(Dt!=0)for(n.eg(t,37),n._f(t,o.Kfb(Dt),Bt),V=0;V<t.R.g;V++)(n.im(t.R,t.C[V])==32||n.im(t.R,t.C[V])==64)&&(n._f(t,o.Kfb(V),Bt),n._f(t,Ct(n.im(t.R,t.C[V])==32?0:1),1));if(t.b&&(t.Q&16)!=0){for(Dt=0,lt=0,v=0;v<t.R.f;v++){for(ut=0,Ht=n.$o(t.R,t.B[v]);Ht<n.Lo(t.R,t.B[v]);Ht++)n.Pm(t.R,n.Zo(t.R,t.B[v],Ht))&&(wt=Ht-n.$o(t.R,t.B[v]),ut|=1<<wt,lt=d.Math.max(lt,wt+1));ut!=0&&++Dt;}if(Dt!=0)for(n.eg(t,38),n._f(t,o.Kfb(Dt),Bt),n._f(t,o.Kfb(lt),3),u=0;u<t.R.f;u++){for(ut=0,Ht=n.$o(t.R,t.B[u]);Ht<n.Lo(t.R,t.B[u]);Ht++)n.Pm(t.R,n.Zo(t.R,t.B[u],Ht))&&(wt=Ht-n.$o(t.R,t.B[u]),ut|=1<<wt);ut!=0&&(n._f(t,o.Kfb(u),Bt),n._f(t,o.Kfb(ut),lt));}}t.R.K&&n.gf(t,39,Bt,uP,4,47),n._f(t,0,1),t.J=n.ag(t);},n.Ag=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(t._=B(o.M7,Y,6,t.R.f,15,1),A=0;A<t.R.f;A++)if(t.db[A]==1||t.db[A]==2){if(Q=t.eb[A],n.Wp(t.R,A))for(u=0;u<n.$o(t.R,A);u++){for(P=n.Zo(t.R,A,u),y=0,L=B(o.Q7,H,6,3,15,1),l=0;l<n.$o(t.R,P);l++)L[y]=n.Zo(t.R,P,l),L[y]!=A&&++y;y==2&&t.f[L[0]]>t.f[L[1]]^t.H[L[0]]<t.H[L[1]]&&(Q=!Q);}else for(u=1;u<n.$o(t.R,A);u++)for(l=0;l<u;l++)i=n.Zo(t.R,A,u),D=n.Zo(t.R,A,l),t.f[i]>t.f[D]&&(Q=!Q),t.H[i]<t.H[D]&&(Q=!Q);t._[A]=t.db[A]==1^Q?1:2;}else t._[A]=t.db[A];for(t.j=B(o.M7,Y,6,t.R.g,15,1),e=0;e<t.R.g;e++)if(t.o[e]==1||t.o[e]==2){for(Q=false,u=0;u<2;u++)if(r=n.Zl(t.R,u,e),n.$o(t.R,r)==3){for(L=B(o.Q7,H,6,2,15,1),y=0,l=0;l<3;l++)n.Zo(t.R,r,l)!=n.Zl(t.R,1-u,e)&&(L[y++]=n.Zo(t.R,r,l));t.f[L[0]]>t.f[L[1]]&&(Q=!Q),t.H[L[0]]<t.H[L[1]]&&(Q=!Q);}t.j[e]=t.o[e]==1^Q?1:2;}else t.j[e]=t.o[e];},n.Bg=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b;for(Q=B(o.Q7,H,6,32,15,1),l=0,r=0;r<t.R.f;r++)(t._[r]==1||t._[r]==2)&&t.bb[r]==A&&(u=t.ab[r],Q[u]<t.f[r]&&(Q[u]==0&&++l,Q[u]=t.f[r]));for(i=0;i<t.R.g;i++)(t.j[i]==1||t.j[i]==2)&&t.n[i]==A&&n.im(t.R,i)==1&&(u=t.k[i],b=d.Math.max(t.f[n.Zl(t.R,0,i)],t.f[n.Zl(t.R,1,i)]),Q[u]<b&&(Q[u]==0&&++l,Q[u]=b));for(D=B(o.M7,Y,6,32,15,1),L=0;L<l;L++){for(T=-1,c=0,y=0;y<32;y++)c<Q[y]&&(c=Q[y],T=y);Q[T]=0,D[T]=L<<24>>24;}for(e=0;e<t.R.f;e++)(t._[e]==1||t._[e]==2)&&t.bb[e]==A&&(t.ab[e]=D[t.ab[e]]);for(P=0;P<t.R.g;P++)(t.j[P]==1||t.j[P]==2)&&t.n[P]==A&&n.im(t.R,P)==1&&(t.k[P]=D[t.k[P]]);},n.Cg=function(t,A,e){var r,P;for(t.U=A,r=0;r<t.R.f;r++)t.f[r]=e[r],t.db[r]=0,t.ib[r]=false;for(P=0;P<t.R.g;P++)t.o[P]=0,t.q[P]=false;},n.Dg=function(t,A){var e,r,P;if(e=n.Zl(t.R,0,A),e>=t.R.f)return  false;if(t.db[e]==1||t.db[e]==2)return  true;if(t.db[e]==3)return  false;if(r=n.Ho(t.R,e),r!=-1)return t.o[r]==1||t.o[r]==2;for(P=0;P<n.$o(t.R,e);P++)if(n.ap(t.R,e,P)==2&&(t.db[n.Zo(t.R,e,P)]==1||t.db[n.Zo(t.R,e,P)]==2))return  true;return  false;},n.Eg=function(t,A){var e;for(e=0;e<n.$o(t.R,A);e++)if(t.o[n._o(t.R,A,e)]==3&&n.ap(t.R,A,e)==1)return  true;return  false;},n.Fg=function(t){var A,e,r,P,i,D,u;for(D=B(o.Q7,H,6,t.U+1,15,1),e=0;e<t.R.f;e++)n.Hl(t.R,e)==0&&(t.db[e]==1?++D[t.f[e]]:t.db[e]==2&&--D[t.f[e]]);for(P=0;P<t.R.g;P++)n.fm(t.R,P)==1&&n.cm(t.R,P)==0&&(t.o[P]==1?(++D[t.f[n.Zl(t.R,0,P)]],++D[t.f[n.Zl(t.R,1,P)]]):t.o[P]==2&&(--D[t.f[n.Zl(t.R,0,P)]],--D[t.f[n.Zl(t.R,1,P)]]));for(u=1;u<=t.U;u++)if(D[u]!=0){if(i=D[u]<0,i){for(A=0;A<t.R.f;A++)n.Hl(t.R,A)==0&&(t.db[A]==1?t.db[A]=2:t.db[A]==2&&(t.db[A]=1));for(r=0;r<t.R.g;r++)n.fm(t.R,r)==1&&n.cm(t.R,r)==0&&(t.o[r]==1?t.o[r]=2:t.o[r]==2&&(t.o[r]=1));}return i;}return  false;},n.Gg=function(t,A){var e;return e=B(o.Q7,H,6,A,15,1),h.Qkb(t,0,e,0,t.length),e;},n.Hg=function(t,A){var e;return e=B(o.vfb,Tt,6,A,16,1),h.Qkb(t,0,e,0,t.length),e;},n.Ig=function(t){var A,e;if(t.$!=null)for(A=0;A<t.R.f;A++)n.hn(t.R,A,t.$[A]);if(t.i!=null)for(e=0;e<t.R.g;e++)n.Fn(t.R,e,t.i[e]);},n.Jg=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(A=0;A<t.R.f;A++)if(t.db[A]==1||t.db[A]==2){if(Q=false,n.Wp(t.R,A))for(u=0;u<n.$o(t.R,A);u++){for(P=n.Zo(t.R,A,u),y=0,L=B(o.Q7,H,6,3,15,1),l=0;l<n.$o(t.R,P);l++)L[y]=n.Zo(t.R,P,l),L[y]!=A&&++y;y==2&&t.f[L[0]]>t.f[L[1]]^L[0]<L[1]&&(Q=!Q);}else for(u=1;u<n.$o(t.R,A);u++)for(l=0;l<u;l++)i=n.Zo(t.R,A,u),D=n.Zo(t.R,A,l),t.f[i]>t.f[D]&&(Q=!Q),i<D&&(Q=!Q);n.un(t.R,A,t.db[A]==1^Q?1:2,t.fb[A]);}else n.un(t.R,A,t.db[A],t.fb[A]);for(e=0;e<t.R.g;e++)if(t.o[e]==1||t.o[e]==2){for(Q=false,u=0;u<2;u++)if(r=n.Zl(t.R,u,e),n.$o(t.R,r)==3){for(L=B(o.Q7,H,6,2,15,1),y=0,l=0;l<3;l++)n.Zo(t.R,r,l)!=n.Zl(t.R,1-u,e)&&(L[y++]=n.Zo(t.R,r,l));t.f[L[0]]>t.f[L[1]]&&(Q=!Q),L[0]<L[1]&&(Q=!Q);}n.Jn(t.R,e,t.o[e]==1^Q?1:2,t.p[e]);}else n.Jn(t.R,e,t.o[e],t.p[e]);},n.Kg=function(t){var A;for(A=0;A<t.R.f;A++)n.yn(t.R,A,t.M[A]);},n.Lg=function(t){var A,e,r;for(A=0;A<t.R.f;A++)!n.wm(t.R,A)&&t.db[A]==3&&n.ln(t.R,A,true);for(e=0;e<t.R.g;e++)t.o[e]==3&&(r=n.fm(t.R,e),r==1?n.ln(t.R,n.Zl(t.R,0,e),true):r==2&&n.Mn(t.R,e,386));},n.Mg=function(t,A){return t<A?t:t-A;},n.Ng=function(t){n.Og.call(this,t,0);},n.Og=function(t,A){this.R=t,this.Q=A,n.bw(this.R,7),this.c=n.Pg(this.R.f),(this.Q&B1)==0&&n.Cf(this),this.jb=(A&64)!=0||n.vm(this.R),this.b=n.np(this.R)==0,(this.Q&B1)==0&&(this.db=B(o.M7,Y,6,this.R.f,15,1),this.fb=B(o.vfb,Tt,6,this.R.f,16,1),this.ib=B(o.vfb,Tt,6,this.R.f,16,1),this.o=B(o.M7,Y,6,this.R.g,15,1),this.q=B(o.vfb,Tt,6,this.R.g,16,1),this.p=B(o.vfb,Tt,6,this.R.g,16,1)),this.d=false,n.Gf(this),(this.Q&B1)==0&&n.Mf(this),n.Lf(this);},n.Pg=function(t){var A;for(A=0;t>0;)t>>=1,++A;return A;},x(34,1,{},n.Ng,n.Og),s.b=false,s.c=0,s.d=false,s.r=false,s.t=0,s.v=0,s.w=0,s.G=false,s.I=0,s.K=false,s.L=false,s.O=0,s.Q=0,s.T=0,s.U=0,s.Z=false,s.jb=false,o.g8=N(34),n.Qg=function(t,A){var e,r,P,i;if(t.d!=A.d)return t.d>A.d?1:-1;for(P=t.a.length,i=A.a.length,e=d.Math.min(P,i),r=0;r<e;r++)if(--P,--i,t.a[P]!=A.a[i])return t.a[P]>A.a[i]?1:-1;return P!=i?P>i?1:-1:t.b!=A.b?t.b>A.b?1:-1:0;},n.Rg=function(t,A){return n.Qg(t,A);},n.Sg=function(){},x(266,1,{},n.Sg),s.Rb=function(t,A){return n.Rg(t,A);},s.mb=function(t){return this===t;},o.Z7=N(266),n.Vg=function(){},x(174,1,{174:1},n.Vg),s.b=0,s.c=0,s.d=0,o.Y7=N(174),n.Wg=function(t,A){return t.c!=A.c?t.c>A.c?1:-1:0;},n.Xg=function(t,A){return n.Wg(t,A);},n.Yg=function(){},x(267,1,{},n.Yg),s.Rb=function(t,A){return n.Xg(t,A);},s.mb=function(t){return this===t;},o._7=N(267),n._g=function(){},x(175,1,{175:1},n._g),s.a=0,s.b=0,s.c=0,o.$7=N(175),n.ah=function(t,A,e){t.b==0&&(++t.c,t.b=63),t.b==63?(t.d[t.c]=Pt(t.d[t.c],e),t.b-=A):t.b>=A?(t.d[t.c]=h1(t.d[t.c],A),t.d[t.c]=Pt(t.d[t.c],e),t.b-=A):(t.d[t.c]=h1(t.d[t.c],t.b),t.d[t.c]=Pt(t.d[t.c],A0(e,A-t.b)),A-=t.b,++t.c,t.b=63-A,t.d[t.c]=Pt(t.d[t.c],J(e,o.Kfb((1<<A)-1))));},n.bh=function(t,A){t.d[t.c]=v1(t.d[t.c],A);},n.dh=function(t,A){var e;for(e=0;e<t.c;e++)if(pt(t.d[e],A.d[e]))return kn(t.d[e],A.d[e])?-1:1;return S1(t.d[t.c],A.d[t.c])?0:kn(t.d[t.c],A.d[t.c])?-1:1;},n.eh=function(t,A){t.a=A,t.c=0,t.b=63,U.Erb(t.d);},n.fh=function(t){this.d=B(o.R7,e0,6,t,14,1);},x(101,1,{101:1,32:1},n.fh),s.Sb=function(t){return n.dh(this,t);},s.a=0,s.b=0,s.c=0,o.a8=N(101),n.hh=function(t,A){return t.b!=A.b?t.b>A.b?-1:1:t.c!=A.c?t.c>A.c?-1:1:0;},n.ih=function(t,A,e){this.b=d.Math.max(t,A),this.c=d.Math.min(t,A),this.a=e;},x(173,1,{173:1,32:1},n.ih),s.Sb=function(t){return n.hh(this,t);},s.a=0,s.b=0,s.c=0,o.b8=N(173),n.kh=function(t,A,e,r){this.a=(w.CAb(A),U.Brb(t,B(o.Q7,H,6,A,15,1),0,A)),this.b=(w.CAb(r),U.Brb(e,B(o.Q7,H,6,r,15,1),0,r));},x(208,1,{208:1},n.kh),o.c8=N(208),n.lh=function(t,A){var e,r;for(r=0;r<t.g[A].length;r++)if(e=t.g[A][r],t.f[e]&&(t.o[e]==1||t.o[e]==2)&&t.k[e]==0)return  true;return  false;},n.mh=function(t,A,e){var r,P,i,D,u;for(P=0,D=0,u=0;u<t.g[A].length;u++)r=t.g[A][u],t.k[r]==e&&(i=1<<t.j[r],(D&i)==0&&(D|=i,++P));return P;},n.nh=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T;for(L=new U.Nxb(),y=0;y<t.i.f;y++)if(n.Qo(t.i,y)<2||n.$o(t.i,y)>2)for(D=1;D<n.$o(t.i,y);D++)for(A=n.Zo(t.i,y,D),u=0;u<D;u++)e=n.Zo(t.i,y,u),n.sh(t,A,e)&&(Q=n.zh(t,A,e),Q&&n.Gr(Q,t.f)&&U.Pwb(L.a,Q,(h.Ghb(),Fe))==null);for(T=0;T<t.i.g;T++)t.c[T]!=0&&(n.fm(t.i,T)!=2||t.c[T]!=2)||(A=n.Zl(t.i,0,T),e=n.Zl(t.i,1,T),n.sh(t,A,e)&&(Q=n.zh(t,A,e),Q&&n.Gr(Q,t.f)&&U.Pwb(L.a,Q,(h.Ghb(),Fe))==null));for(t.g=B(o.Q7,E1,7,L.a.c,0,2),t.e=B(o.vfb,Tt,6,t.i.f,16,1),i=0,l=(P=new U.fxb(new U.lxb(new U.sqb(L.a).a).b),new U.Bqb(P));U.apb(l.a.a);)for(Q=(r=U.dxb(l.a),r.hd()),t.g[i++]=Q.b,D=0;D<Q.b.length;D++)t.e[Q.b[D]]=true;},n.oh=function(t,A,e,r){var P,i,D,u,Q,l,L;for(P=B(o.Q7,H,6,n.$o(t.i,e),15,1),u=0,D=0;D<n.$o(t.i,e);D++)P[u]=n.Zo(t.i,e,D),!r[P[u]]&&n.sh(t,A,P[u])&&++u;if(u==0)return  -1;if(u==1)return P[0];for(Q=-1,l=qt,i=0;i<u;i++)L=n.Bp(t.i,A,P[i],qt,r),L<l&&(l=L,Q=P[i]);return Q;},n.ph=function(t,A){return t.f[A]&&(t.o[A]==1||t.o[A]==2);},n.qh=function(t,A){return t.e[A];},n.rh=function(t){var A,e;for(e=true,A=0;A<t.i.f;A++)if(t.o[A]!=0&&!t.e[A]){e=false;break;}return e;},n.sh=function(t,A,e){var r,P,i,D,u;if(A==e||t.a[A]!=t.a[e]||t.o[A]!=0&&(t.o[A]==3||t.o[e]==3||t.p[A]^t.o[A]!==t.o[e]||t.k[A]!==t.k[e]||t.j[A]!==t.j[e])||(r=n.Xo(t.i,A,e),r!=-1&&(n.fm(t.i,r)==1&&t.c[r]!=0||n.fm(t.i,r)==2&&t.c[r]==1)))return  false;if(n.Qo(t.i,A)==1&&!n.Tp(t.i,A)){for(P=-1,u=0;u<n.$o(t.i,A);u++)if(n.Zo(t.i,A,u)!=e&&n.ap(t.i,A,u)==2){P=n._o(t.i,A,u);break;}for(i=-1,D=0;D<n.$o(t.i,e);D++)if(n.Zo(t.i,e,D)!=A&&n.ap(t.i,e,D)==2){i=n._o(t.i,e,D);break;}if(P!=-1&&t.c[P]!=0&&t.d[P]^t.c[P]===t.c[i])return  false;}return  true;},n.th=function(t,A,e,r,P){var i,D,u,Q,l,L;for(Q=null,i=null,L=0;L<t.g[A].length;L++)D=t.g[A][L],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?i=n.Bh(i,(P[D]<<16)+D):t.k[D]==r&&t.j[D]==e&&(Q=n.Bh(Q,(P[D]<<16)+D)));if(u=n.Ph(Q,i),u==0)return  false;if(u<0)for(l=0;l<t.g[A].length;l++)D=t.g[A][l],t.f[D]&&(t.o[D]==1||t.o[D]==2)&&(t.k[D]==0?(t.k[D]=r<<24>>24,t.j[D]=e<<24>>24):t.k[D]==r&&t.j[D]==e&&(t.k[D]=0,t.j[D]=-1));return  true;},n.uh=function(t,A){var e,r,P,i,D,u;if(!t.b)return  false;for(P=false,i=t.b.a.length-1;i>=0;i--)if(r=false,D=U.xj(t.b,i),D.a==2?r=n.th(t,D.b,D.c,D.d,A):D.a==1&&(r=n.yh(t,D.b,A)),r){for(U.Fj(t.b,D),u=0;u<t.g[D.b].length;u++)e=t.g[D.b][u],t.n[e]=false;P=true;}return P;},n.vh=function(t){var A,e,r,P,i,D,u,Q;if(t.g!=null)for(D=new n.Oh(t),t.b=new U.Mj(),P=0;P<t.g.length;P++)r=n.Gh(D,P),r==0?(n.Ch(D,P),u=n.mh(t,P,2),A=n.mh(t,P,1),e=n.lh(t,P),u==1&&A==1&&!e&&(n.xh(t,P,D.a+D.f++),U.sj(t.b,new n.jk(P,1,-1,-1))),u>0?(e&&(n.wh(t,P,D.i+D.g++,2),++u),U.sj(t.b,new n.jk(P,1,-1,-1))):A>0?(e&&n.wh(t,P,D.a+D.f++,1),U.sj(t.b,new n.jk(P,1,-1,-1))):e&&(n.wh(t,P,D.a+D.f++,1),U.sj(t.b,new n.jk(P,1,-1,-1)))):r==1&&(n.lh(t,P)?(i=n.Fh(D,P),Q=n.Hh(D,P),U.sj(t.b,new n.jk(P,2,i,Q))):(n.Ch(D,P),U.sj(t.b,new n.jk(P,1,-1,-1))));},n.wh=function(t,A,e,r){var P,i;for(i=0;i<t.g[A].length;i++)P=t.g[A][i],t.f[P]&&(t.o[P]==1||t.o[P]==2)&&t.k[P]==0&&(t.k[P]=r<<24>>24,t.j[P]=e<<24>>24);},n.xh=function(t,A,e){var r,P;for(P=0;P<t.g[A].length;P++)r=t.g[A][P],t.k[r]==2&&(t.k[r]=1,t.j[r]=e<<24>>24);},n.yh=function(t,A,e){var r,P,i,D,u,Q,l,L;for(i=t.g[A],P=1,Q=0;Q<i.length;Q++)if(r=i[Q],t.f[r]&&t.k[r]==2){P=2;break;}for(D=B(o.Q7,E1,7,32,0,2),l=0;l<i.length;l++)r=i[l],t.f[r]&&t.k[r]==P&&(D[t.j[r]]=n.Bh(D[t.j[r]],(e[r]<<16)+r));for(L=0;L<32;L++)D[L]!=null&&w.lAb(D[L],w.SAb(T0(U.lsb.prototype.ld,U.lsb,[])));if(U.Vrb(D,new n.Qh()),n.Ph(D[0],D[1])==0)return  false;for(u=0;u<D[0].length;u++)r=D[0][u]&e1,t.k[r]=0,t.j[r]=-1;return  true;},n.zh=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;for(Q=B(o.Q7,H,6,t.i.f,15,1),v=B(o.Q7,H,6,t.i.f,15,1),b=B(o.vfb,Tt,6,t.i.f,16,1),l=B(o.vfb,Tt,6,t.i.f,16,1),S=new n.Hr(t.i.f),Q[0]=A,v[A]=e,v[e]=-2,S.a[A]=true,S.a[e]=true,i=0,L=0;i<=L;){if(D=Q[i],v[D]==D){for(y=0;y<n.$o(t.i,D);y++)if(r=n.Zo(t.i,D,y),!S.a[r]){if(n.ap(t.i,D,y)==2&&n.Tl(t.i,r)<10)Q[++L]=r,v[r]=r,l[r]=l[D]||n.Qo(t.i,r)==2,b[r]=l[D]&&!b[D],S.a[r]=true;else if(l[D]&&b[D]){if(p=n.oh(t,r,v[D],S.a),p==-1)return null;Q[++L]=r,v[r]=p,v[p]=-2,l[r]=false,S.a[r]=true,S.a[p]=true;}else if(n.gq(t.i,n._o(t.i,D,y))&&(Q[++L]=r,v[r]=r,l[r]=false,S.a[r]=true,(n.Tl(t.i,r)==6&&n.Qo(t.i,r)==0||n.Tl(t.i,r)==7&&n.Bl(t.i,r)==1||n.Tl(t.i,r)==14||n.Tl(t.i,r)==15&&n.$o(t.i,r)>2||n.Tl(t.i,r)==16&&n.$o(t.i,r)>2)&&n.$o(t.i,r)>2)){for(u=false,g=1;g<n.$o(t.i,r);g++)if(E=n.Zo(t.i,r,g),!S.a[E])for(k=0;k<g;k++)q=n.Zo(t.i,r,k),S.a[q]||n.sh(t,E,q)&&(Q[++L]=E,v[E]=q,v[q]=-2,l[E]=false,S.a[E]=true,S.a[q]=true,u=true);if(!u)return null;}}}else {for(P=B(o.vfb,Tt,6,n.$o(t.i,D),16,1),T=0;T<n.$o(t.i,D);T++)if(r=n.Zo(t.i,D,T),S.a[r])P[T]=v[r]==r;else for(g=0;g<n.$o(t.i,r);g++)if(n.Zo(t.i,r,g)==v[D]){P[T]=true;break;}for(c=0;c<n.$o(t.i,D);c++)if(P[c])if(r=n.Zo(t.i,D,c),S.a[r]){if(n.Xo(t.i,r,v[D])==-1)return null;}else Q[++L]=r,v[r]=r,b[r]=false,l[r]=true,S.a[r]=true;for(y=0;y<n.$o(t.i,D);y++)if(!P[y]&&(r=n.Zo(t.i,D,y),!S.a[r])){if(p=n.oh(t,r,v[D],S.a),p==-1)return null;Q[++L]=r,v[r]=p,v[p]=-2,l[r]=false,S.a[r]=true,S.a[p]=true;}}++i;}return S;},n.Ah=function(t,A,e,r,P,i,D,u,Q,l){this.i=t,this.a=A,this.f=e,this.o=r,this.c=P,this.k=i,this.j=D,this.p=u,this.d=Q,this.n=l,n.nh(this);},n.Bh=function(t,A){var e,r;for(r=B(o.Q7,H,6,t==null?1:t.length+1,15,1),e=0;e<r.length-1;e++)r[e]=t[e];return r[r.length-1]=A,r;},x(265,1,{},n.Ah),o.e8=N(265),n.Ch=function(t,A){var e,r,P,i,D,u;for(P=0;P<t.b;P++)if(t.e[P][A]&&t.c[P]!=-3){for(r=0;r<=t.j.g.length;r++)if(r!=A&&t.e[P][r])for(t.e[P][A]=false,u=P<t.a?P:P<t.b?P-t.a:-1,D=n.Mh(t,P<t.a?1:P<t.b?2:0),i=0;i<t.j.g[A].length;i++)e=t.j.g[A][i],n.ph(t.j,e)&&t.j.j[e]==u&&(t.j.j[e]=D<<24>>24);}},n.Dh=function(t,A,e){var r,P,i,D,u;for(u=false,D=1,A[e]=D,r=true;r;){for(r=false,P=0;P<t.b;P++)if(A[P]==D)for(i=0;i<t.b;i++)A[i]==0&&n.Lh(t,P,i)&&(t.c[i]==-2?(A[i]=D+1,r=true):t.c[i]!=t.c[e]&&(A[i]=D+1,u=true));++D;}return u;},n.Eh=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b;for(Q=r+1;Q<t.j.g.length;Q++)if(Q!=r&&t.e[A][Q]&&t.e[e][Q])return D=B(o.Q7,H,6,2,15,1),D[0]=e,D[1]=A,D;for(b=B(o.Q7,H,6,t.b,15,1),L=B(o.Q7,H,6,t.b,15,1),l=B(o.Q7,H,6,t.b,15,1),i=0,y=0,l[0]=A,L[A]=1;i<=y;){for(T=0;T<t.d[l[i]].length;T++){if(P=t.d[l[i]][T],P==e){if(i==0)continue;for(u=L[l[i]]+1,D=B(o.Q7,H,6,u,15,1),D[0]=P,D[1]=l[i],c=2;c<u;c++)D[c]=b[D[c-1]];return D;}L[P]==0&&t.c[P]!=-3&&(L[P]=L[l[i]]+1,l[++y]=P,b[P]=l[i]);}++i;}return null;},n.Fh=function(t,A){var e;for(e=0;e<t.b;e++)if(t.e[e][A]&&t.c[e]==-3)return e<t.a?e:e<t.b?e-t.a:-1;return  -1;},n.Gh=function(t,A){var e,r;for(e=0,r=0;r<t.b;r++)t.e[r][A]&&t.c[r]==-3&&++e;return e;},n.Hh=function(t,A){var e;for(e=0;e<t.b;e++)if(t.e[e][A]&&t.c[e]==-3)return e<t.a?1:e<t.b?2:0;return  -1;},n.Ih=function(t,A){return A<t.a?A:A<t.b?A-t.a:-1;},n.Jh=function(t,A){return A<t.a?1:A<t.b?2:0;},n.Kh=function(t,A){var e,r;return r=t.j.k[A],e=t.j.j[A],r==0?t.b:r==1?e:t.a+e;},n.Lh=function(t,A,e){var r;for(r=0;r<t.j.g.length;r++)if(t.e[A][r]&&t.e[e][r])return  true;return  false;},n.Mh=function(t,A){return A==1?t.a+t.f++:t.i+t.g++;},n.Nh=function(t,A){var e,r,P,i,D,u,Q,l,L,y;for(L=qt,Q=-1,y=-1,l=-1,r=0;r<t.j.i.f;r++)if(n.ph(t.j,r)&&t.j.k[r]!=0)for(u=0;u<A.length;u++)P=n.Ih(t,A[u]),i=n.Jh(t,A[u]),t.j.k[r]==i&&t.j.j[r]==P&&L>t.j.a[r]+(i==1?H1:0)&&(L=t.j.a[r]+(i==1?H1:0),Q=P,y=i,l=A[u]);for(e=0;e<t.j.i.f;e++)n.ph(t.j,e)&&t.j.k[e]==y&&t.j.j[e]==Q&&(t.j.k[e]=0,t.j.j[e]=-1);for(D=0;D<t.j.g.length;D++)t.e[l][D]=false;},n.Oh=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(this.j=t,r=0;r<t.i.f;r++)t.f[r]&&(t.o[r]==1||t.o[r]==2)&&(t.k[r]==1?this.a<=t.j[r]&&(this.a=1+t.j[r]):t.k[r]==2&&this.i<=t.j[r]&&(this.i=1+t.j[r]));for(this.b=this.a+this.i,this.e=U1(o.vfb,[o1,Tt],[12,6],16,[this.b+1,t.g.length+1],2),P=0;P<t.i.f;P++)t.f[P]&&(t.o[P]==1||t.o[P]==2)&&!t.e[P]&&(this.e[n.Kh(this,P)][t.g.length]=true);for(Q=0;Q<t.g.length;Q++)for(k=0;k<t.g[Q].length;k++)e=t.g[Q][k],t.f[e]&&(t.o[e]==1||t.o[e]==2)&&(this.e[n.Kh(this,e)][Q]=true);for(this.d=B(o.Q7,E1,7,this.b,0,2),l=0;l<t.g.length;l++)for(c=1;c<this.b;c++)if(this.e[c][l])for(b=0;b<c;b++)this.e[b][l]&&(this.d[c]=n.Bh(this.d[c],b),this.d[b]=n.Bh(this.d[b],c));for(this.c=B(o.Q7,H,6,this.b+1,15,1),T=0;T<this.b;T++)this.e[T][t.g.length]?this.c[T]=-1:this.c[T]=-2;for(L=0;L<t.g.length;L++)if(this.e[this.b][L])for(y=0;y<this.b;y++)this.e[y][L]&&this.c[y]!=L&&(this.c[y]==-2?this.c[y]=L:this.c[y]=-3);for(A=0;A<this.b;A++)if(this.c[A]>=-1&&(i=B(o.Q7,H,6,this.b,15,1),n.Dh(this,i,A)))for(y=0;y<this.b;y++)i[y]!=0&&(this.c[y]=-3);for(u=0;u<t.g.length-1;u++)for(c=1;c<this.b;c++)if(this.e[c][u]&&this.c[c]!=-3){for(b=0;b<c;b++)if(this.e[b][u]&&this.c[b]!=-3&&(D=n.Eh(this,c,b,u),D!=null)){for(g=0;g<D.length;g++)this.c[D[g]]=-3;n.Nh(this,D);break;}}},x(366,1,{},n.Oh),s.a=0,s.b=0,s.f=0,s.g=0,s.i=0,o.d8=N(366),n.Ph=function(t,A){var e,r;if(t==null)return A==null?0:1;if(A==null)return  -1;for(e=d.Math.min(t.length,A.length),r=0;r<e;r++)if((t[r]&fA)!=(A[r]&fA))return (t[r]&fA)<(A[r]&fA)?-1:1;return t.length==A.length?0:t.length<A.length?-1:1;},n.Qh=function(){},x(367,1,{},n.Qh),s.Rb=function(t,A){return n.Ph(t,A);},s.mb=function(t){return this===t;},o.f8=N(367),n.Th=function(t,A){switch(A){case 0:return n.Uh(t,false);case 1:return n.Wh(t);case 2:return n.Vh(t);case 3:return n.Yh(t);case 4:return n.Xh(t);}return null;},n.Uh=function(t,A){try{return A?(t=n.fw(t),n.zq(t,!0),n.Tr(t),n.rg(new n.Ng(t))):n.rg(new n.Ng(t));}catch(e){if(e=C1(e),Zt(e,27))return h.Pkb(),null;throw st(e);}},n.Vh=function(t){var A,e;try{for(t=n.fw(t),n.pw(t),A=t.r,e=0;e<A;e++)t.H[e]=1,t.T=0;return n.rg(new n.Ng(t));}catch(r){if(r=C1(r),Zt(r,27))return h.Pkb(),null;throw st(r);}},n.Wh=function(t){try{return t=n.fw(t),n.pw(t),n.rg(new n.Ng(t));}catch(A){if(A=C1(A),Zt(A,27))return h.Pkb(),null;throw st(A);}},n.Xh=function(t){var A;try{return t=n.fw(t),n.pw(t),A=n.Hw(new n.Kw(t)),n.rg(new n.Og(A,8));}catch(e){if(e=C1(e),Zt(e,27))return h.Pkb(),null;throw st(e);}},n.Yh=function(t){var A;try{return A=n.Hw(new n.Kw(t)),n.rg(new n.Og(A,8));}catch(e){if(e=C1(e),Zt(e,27))return h.Pkb(),null;throw st(e);}},n.Zh=function(t,A,e,r){t.a+=A,t.b+=e,t.c+=r;},n.$h=function(t,A){return t.a+=A.a,t.b+=A.b,t.c+=A.c,t;},n._h=function(t,A){return new n.oi(t.a+A.a,t.b+A.b,t.c+A.c);},n.ai=function(t,A){return t.a=(t.a+A.a)/2,t.b=(t.b+A.b)/2,t.c=(t.c+A.c)/2,t;},n.bi=function(t,A){return t.a!=A.a?t.a<A.a?-1:1:t.b!=A.b?t.b<A.b?-1:1:t.c!=A.c?t.c<A.c?-1:1:0;},n.ci=function(t,A){return new n.oi(t.b*A.c-t.c*A.b,-(t.a*A.c-t.c*A.a),t.a*A.b-t.b*A.a);},n.di=function(t,A){return d.Math.sqrt((A.a-t.a)*(A.a-t.a)+(A.b-t.b)*(A.b-t.b)+(A.c-t.c)*(A.c-t.c));},n.ei=function(t,A){var e;return A==null||!Zt(A,29)?false:(e=A,d.Math.abs(e.a-t.a)+d.Math.abs(e.b-t.b)+d.Math.abs(e.c-t.c)<1e-6);},n.fi=function(t,A){var e,r,P;return r=t.a*t.a+t.b*t.b+t.c*t.c,P=A.a*A.a+A.b*A.b+A.c*A.c,r==0||P==0||(e=(t.a*A.a+t.b*A.b+t.c*A.c)/d.Math.sqrt(r*P),e>=1)?0:e<=-1?A1:d.Math.acos(e);},n.gi=function(t,A){var e,r,P;return e=t.a,r=t.b,P=t.c,t.a=e*A[0][0]+r*A[1][0]+P*A[2][0],t.b=e*A[0][1]+r*A[1][1]+P*A[2][1],t.c=e*A[0][2]+r*A[1][2]+P*A[2][2],t;},n.hi=function(t,A){return t.a*=A,t.b*=A,t.c*=A,t;},n.ii=function(t,A,e,r){t.a=A,t.b=e,t.c=r;},n.ji=function(t,A){return n.ii(t,A.a,A.b,A.c),t;},n.ki=function(t,A){return t.a-=A.a,t.b-=A.b,t.c-=A.c,t;},n.li=function(t,A){return new n.oi(t.a-A.a,t.b-A.b,t.c-A.c);},n.mi=function(t){var A;return A=d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),A==0?(h.Pkb(),t.a=1,t.b=0,t.c=0,t):(t.a/=A,t.b/=A,t.c/=A,t);},n.ni=function(){},n.oi=function(t,A,e){this.a=t,this.b=A,this.c=e;},n.pi=function(t){n.oi.call(this,t.a,t.b,t.c);},x(29,1,{29:1,4:1,32:1},n.ni,n.oi,n.pi),s.Sb=function(t){return n.bi(this,t);},s.mb=function(t){return n.ei(this,t);},s.pb=function(){var t;return t=new o0.Hnb("0.00"),"["+o0.Dnb(t,this.a)+", "+o0.Dnb(t,this.b)+", "+o0.Dnb(t,this.c)+"]";},s.a=0,s.b=0,s.c=0,o.h8=N(29),n.ti=function(t,A){A.Nb(t.c),n.le(A,t.a,t.b);},n.ui=function(t,A){A.c*=t.c,A.a=A.a*t.c+t.a,A.b=A.b*t.c+t.b;},n.vi=function(t,A){n.dn(A,t.c),n._n(A,t.a,t.b);},n.wi=function(t,A){A.a=A.a*t.c+t.a,A.b=A.b*t.c+t.b;},n.xi=function(t,A){A.c=A.c*t.c+t.a,A.d=A.d*t.c+t.b,A.b*=t.c,A.a*=t.c;},n.yi=function(t){t.a=0,t.b=0,t.c=1;},n.zi=function(t){return t.c==1&&t.a==0&&t.b==0;},n.Ai=function(t,A){return A*t.c+t.a;},n.Bi=function(t,A){return A*t.c+t.b;},n.Ci=function(){n.yi(this);},n.Di=function(t,A){this.c=1,this.a=t,this.b=A;},n.Ei=function(t,A,e,r){var P,i,D,u;n.yi(this),A?(r&Ye)==0?ht.kO(A,t.c,t.d)&&ht.kO(A,t.c+t.b,t.d+t.a)||((t.b>A.b||t.a>A.a)&&(i=A.b/t.b,u=A.a/t.a,this.c=d.Math.min(i,u)),t.c*this.c<A.c?this.a=A.c-t.c*this.c:(t.c+t.b)*this.c>A.c+A.b&&(this.a=A.c+A.b-(t.c+t.b)*this.c),t.d*this.c<A.d?this.b=A.d-t.d*this.c:(t.d+t.a)*this.c>A.d+A.a&&(this.b=A.d+A.a-(t.d+t.a)*this.c)):(i=A.b/t.b,u=A.a/t.a,D=r&e1,D==0?D=24:(r&r0)!=0&&(D/=256),P=D/e,this.c=d.Math.min(P,d.Math.min(i,u)),this.a=A.c+A.b/2-this.c*(t.c+t.b/2),this.b=A.d+A.a/2-this.c*(t.d+t.a/2)):(r&H1)!=0&&(D=(r&e1)!=0?r&e1:24,this.c=D/e);},x(78,1,{},n.Ci,n.Di,n.Ei),s.pb=function(){return "DepictorTransformation Offset: "+this.a+","+this.b+" Scaling: "+this.c;},s.a=0,s.b=0,s.c=0,o.i8=N(78),n.Gi=function(t){var A,e,r,P;return !h.akb(w.SAb(t).substr(0,21),iP)||(r=w.SAb(Ze).length,!h.akb(w.SAb(t).substr(w.SAb(t).length-r,r),Ze))||(e=h.ikb(t,X1(34),21),e==-1)?null:(P=(w.MAb(21,e,w.SAb(t).length),w.SAb(t).substr(21,e-21)),A=h.rkb(t,21+w.SAb(P).length+1,w.SAb(t).length-17),h.akb(P,"arrow")?new At.HH(A):h.akb(P,"text")?new n.bx(A):null);},U.ysb=function(t){return new U.Awb(t);},U.zsb=function(t){return new mt.Vzb(t.Ub());},U.Hi=function(t,A){var e,r,P;for(w.GAb(A),e=false,P=A.Tb();P.ed();)r=P.fd(),e=e|t.add(r);return e;},U.Ii=function(t,A,e){var r,P;for(P=t.Tb();P.ed();)if(r=P.fd(),F1(A)===F1(r)||A!=null&&h.Oc(A,r))return e&&P.gd(),true;return  false;},U.Ji=function(t,A){var e,r;for(w.GAb(A),r=A.Tb();r.ed();)if(e=r.fd(),!t.contains(e))return  false;return  true;},U.Ki=function(t,A){var e,r,P;for(w.GAb(A),e=false,r=t.Tb();r.ed();)P=r.fd(),A.contains(P)&&(r.gd(),e=true);return e;},U.Li=function(t,A){var e,r,P;for(w.GAb(A),e=false,r=t.Tb();r.ed();)P=r.fd(),A.contains(P)||(r.gd(),e=true);return e;},U.Mi=function(t,A){var e,r,P,i;for(i=t.size(),A.length<i&&(A=w.mAb(new Array(i),A)),P=A,r=t.Tb(),e=0;e<i;++e)a1(P,e,r.fd());return A.length>i&&o.z6(A,i,null),A;},U.Ni=function(t){var A,e,r;for(r=new U.Fwb(", ","[","]"),e=t.Tb();e.ed();)A=e.fd(),U.Cwb(r,F1(A)===F1(t)?"(this Collection)":A==null?PA:Xn(A));return r.a?w.SAb(r.e).length==0?r.a.a:r.a.a+(""+r.e):r.c;},x(43,1,_e),s.Ub=function(){return new U.Awb(this);},s.Vb=function(){return new mt.Vzb(this.Ub());},s.add=function(t){throw st(new h.Tkb("Add not supported on this collection"));},s.addAll=function(t){return U.Hi(this,t);},s.clear=function(){var t;for(t=this.Tb();t.ed();)t.fd(),t.gd();},s.contains=function(t){return U.Ii(this,t,false);},s.containsAll=function(t){return U.Ji(this,t);},s.isEmpty=function(){return this.size()==0;},s.remove=function(t){return U.Ii(this,t,true);},s.removeAll=function(t){return U.Ki(this,t);},s.retainAll=function(t){return U.Li(this,t);},s.toArray=function(){return this.Wb(B(o.Icb,R1,1,this.size(),5,1));},s.Wb=function(t){return U.Mi(this,t);},s.pb=function(){return U.Ni(this);},o.Ycb=N(43),U.svb=function(t){return new U.Awb(t);},U.tvb=function(t){var A;return mt.Tzb(t.Vb(),(A=mt.bzb(new mt.izb(),mt.Ryb(new mt.gzb(),new mt.ezb(),new mt.Izb(),j(F(o.oeb,1),y0,96,0,[(mt.Vyb(),mt.Tyb)]))),mt.Zyb(A,new mt.lzb())));},U.aj=function(t,A){var e,r;for(e=0,r=t.size();e<r;++e)if(U.Hvb(A,t.getAtIndex(e)))return e;return  -1;},U.bj=function(t,A,e){return new U.lpb(t,A,e);},x(95,43,z8),s.Ub=function(){return new U.Awb(this);},s.addAtIndex=function(t,A){throw st(new h.Tkb("Add not supported on this list"));},s.add=function(t){return this.addAtIndex(this.size(),t),true;},s.addAllAtIndex=function(t,A){var e,r,P;for(w.GAb(A),e=false,P=A.Tb();P.ed();)r=P.fd(),this.addAtIndex(t++,r),e=true;return e;},s.clear=function(){this.Xb(0,this.size());},s.mb=function(t){var A,e,r,P,i;if(t===this)return  true;if(!Zt(t,83)||(i=t,this.size()!=i.size()))return  false;for(P=i.Tb(),e=this.Tb();e.ed();)if(A=e.fd(),r=P.fd(),!(F1(A)===F1(r)||A!=null&&h.Oc(A,r)))return  false;return  true;},s.ob=function(){return U.Bsb(this);},s.indexOf=function(t){return U.aj(this,t);},s.Tb=function(){return new U.dpb(this);},s.lastIndexOf=function(t){var A;for(A=this.size()-1;A>-1;--A)if(U.Hvb(t,this.getAtIndex(A)))return A;return  -1;},s.removeAtIndex=function(t){throw st(new h.Tkb("Remove not supported on this list"));},s.Xb=function(t,A){var e,r;for(r=new U.jpb(this,t),e=t;e<A;++e)w.EAb(r.a<r.c.size()),r.c.getAtIndex(r.b=r.a++),U.cpb(r);},s.setAtIndex=function(t,A){throw st(new h.Tkb("Set not supported on this list"));},s.subList=function(t,A){return U.bj(this,t,A);},o.ddb=N(95);function Yu(f){return f==null?false:f.$implements__java_lang_Cloneable||Array.isArray(f);}U.qj=function(t){t.a=B(o.Icb,R1,1,0,5,1);},U.rj=function(t,A,e){w.IAb(A,t.a.length),w.fAb(t.a,A,e);},U.sj=function(t,A){return w.iAb(t.a,A),true;},U.tj=function(t,A,e){var r,P;return w.IAb(A,t.a.length),r=e.toArray(),P=r.length,P==0?false:(w.gAb(t.a,A,r),true);},U.uj=function(t,A){var e,r;return e=A.toArray(),r=e.length,r==0?false:(w.gAb(t.a,t.a.length,e),true);},U.vj=function(t){w.kAb(t.a,0);},U.wj=function(t,A){return U.zj(t,A,0)!=-1;},U.xj=function(t,A){return w.FAb(A,t.a.length),t.a[A];},U.yj=function(t,A){return U.zj(t,A,0);},U.zj=function(t,A,e){for(;e<t.a.length;++e)if(U.Hvb(A,t.a[e]))return e;return  -1;},U.Aj=function(t){return t.a.length==0;},U.Bj=function(t){return new U.rrb(t);},U.Cj=function(t,A){return U.Dj(t,A,t.a.length-1);},U.Dj=function(t,A,e){for(;e>=0;--e)if(U.Hvb(A,t.a[e]))return e;return  -1;},U.Ej=function(t,A){var e;return e=(w.FAb(A,t.a.length),t.a[A]),w.jAb(t.a,A,1),e;},U.Fj=function(t,A){var e;return e=U.zj(t,A,0),e==-1?false:(U.Ej(t,e),true);},U.Gj=function(t,A,e){var r;w.JAb(A,e,t.a.length),r=e-A,w.jAb(t.a,A,r);},U.Hj=function(t,A,e){var r;return r=(w.FAb(A,t.a.length),t.a[A]),t.a[A]=e,r;},U.Ij=function(t){return t.a.length;},U.Jj=function(t,A){U.Urb(t.a,t.a.length,A);},U.Kj=function(t){return w.cAb(t.a);},U.Lj=function(t,A){var e,r;for(r=t.a.length,A.length<r&&(A=w.mAb(new Array(r),A)),e=0;e<r;++e)a1(A,e,t.a[e]);return A.length>r&&o.z6(A,r,null),A;},U.Mj=function(){U.qj(this);},U.Nj=function(t){U.qj(this),w.AAb(t>=0,"Initial capacity must not be negative");},x(20,95,We,U.Mj,U.Nj),s.addAtIndex=function(t,A){U.rj(this,t,A);},s.add=function(t){return U.sj(this,t);},s.addAllAtIndex=function(t,A){return U.tj(this,t,A);},s.addAll=function(t){return U.uj(this,t);},s.clear=function(){U.vj(this);},s.contains=function(t){return U.wj(this,t);},s.getAtIndex=function(t){return U.xj(this,t);},s.indexOf=function(t){return U.yj(this,t);},s.isEmpty=function(){return U.Aj(this);},s.Tb=function(){return U.Bj(this);},s.lastIndexOf=function(t){return U.Cj(this,t);},s.removeAtIndex=function(t){return U.Ej(this,t);},s.remove=function(t){return U.Fj(this,t);},s.Xb=function(t,A){U.Gj(this,t,A);},s.setAtIndex=function(t,A){return U.Hj(this,t,A);},s.size=function(){return U.Ij(this);},s.toArray=function(){return U.Kj(this);},s.Wb=function(t){return U.Lj(this,t);},o.vdb=N(20),n.ek=function(t){var A,e,r;for(r=new h.Kkb(),e=new U.rrb(t);e.a<e.c.a.length;)A=U.qrb(e),h.Ikb(h.Ikb(r,iP+A.Lb()+'"'+A.Kb()+Ze),d1);return r.a;},n.fk=function(){U.Mj.call(this);},n.gk=function(t){var A,e;U.Mj.call(this);try{if(t)for(e=new U.rrb(t);e.a<e.c.a.length;)A=U.qrb(e),U.sj(this,A.Gb());}catch(r){if(r=C1(r),!Zt(r,19))throw st(r);}},n.hk=function(t){var A,e,r;if(U.Mj.call(this),!(t==null||w.SAb(t).length==0))for(A=0,e=h.hkb(t,X1(10));e!=-1;)r=n.Gi((w.MAb(A,e,w.SAb(t).length),w.SAb(t).substr(A,e-A))),r&&w.iAb(this.a,r),A=e+1,e=h.ikb(t,X1(10),A);},x(103,20,We,n.fk,n.gk,n.hk),s.pb=function(){return n.ek(this);},o.j8=N(103),n.jk=function(t,A,e,r){this.b=t,this.a=A,this.c=e,this.d=r;},x(109,1,{109:1},n.jk),s.a=0,s.b=0,s.c=0,s.d=0,o.k8=N(109),n.kk=function(t){var A,e,r,P,i,D;if(t.j!=0)return t.j;if(t.i&&n.Tl(t.e,t.a)!=15&&n.Tl(t.e,t.a)!=16){for(D=0;D<n.Lo(t.e,t.a);D++)if(i=n._o(t.e,t.a,D),n.Sm(t.e,i,t.a))return n.Zo(t.e,t.a,D)==t.b?t.j=n.im(t.e,i)==257?3:1:t.j=n.im(t.e,i)==257?1:3,t.j;}return A=n.Yl(t.e,t.a,t.g),r=n.Yl(t.e,t.a,t.b),r<A&&(r+=N1),n.Lo(t.e,t.a)==2?(e=r-A,e>3.0915926535897933&&e<3.191592653589793?(t.j=-1,t.j):(t.j=e<A1?4:2,t.j)):(P=n.Yl(t.e,t.a,t.d),P<A&&(P+=N1),t.j=P<r?2:4,t.j);},n.lk=function(t,A,e,r){var P,i,D,u;for(this.e=t,this.g=e,this.a=r,D=-1,u=0;u<n.Lo(this.e,this.a);u++){if(P=n.Zo(this.e,this.a,u),i=n._o(this.e,this.a,u),P==this.g){n.im(this.e,i)==386&&(this.j=-1);continue;}n.Sm(this.e,i,this.a)&&(this.i&&(t.u[r]|=H1),this.i=true),D==A[P]?(this.d=P,this.f=true,this.c=n.gq(this.e,i)):D<A[P]?(D=A[P],this.d=this.b,this.b=P):this.d=P;}},x(116,1,{},n.lk),s.a=0,s.b=0,s.c=false,s.d=0,s.f=false,s.g=0,s.i=false,s.j=0,o.l8=N(116),n.mk=function(t,A,e,r){this.c=t,this.b=A,this.d=e,this.a=r;},x(18,1,{18:1},n.mk),s.pb=function(){return this.d;},s.a=0,s.c=0,o.m8=N(18),n.ok=function(t){var A,e,r,P,i;if(P=new U.Uvb(new n.Hk()),i=new U.Rvb(),t.k!=null)for(r=0;r<t.k.length;r++)for(A=0;A<t.k[r].r;A++)U.Nvb(P,w.SAb(t.e[r].R.c*n.dm(t.k[r],A))),U.Nvb(i,U.Ovb(P)),P.b.a.length<i.b.a.length&&U.Nvb(P,U.Ovb(i));return e=P.b.a.length+i.b.a.length,e==0?n.pk(t):t.p.c*(e%2==0?(h.Rib(P.b.a.length==0?null:U.xj(P.b,0))+h.Rib(i.b.a.length==0?null:U.xj(i.b,0)))/2:h.Rib(P.b.a.length==0?null:U.xj(P.b,0)));},n.pk=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g;if(y=s0,T=QP,b=s0,g=QP,P=0,t.k!=null)for(u=0;u<t.k.length;u++)for(e=0;e<t.k[u].q;e++)L=n.Ai(t.e[u].R,n.Dl(t.k[u],e).a),c=n.Bi(t.e[u].R,n.Dl(t.k[u],e).b),y=d.Math.min(y,L),T=d.Math.max(T,L),b=d.Math.min(b,c),g=d.Math.max(g,c),++P;return P<=1?t.d:(i=T-y,D=g-b,Q=(i+D)/2,l=3*P,A=i*D,r=Q/(1-l),d.Math.sqrt(r*r-A/(1-l))-r);},n.qk=function(t){var A;if(t.p=new n.Ci(),t.k!=null)for(t.e=B(o.Sab,R1,105,t.k.length,0,1),A=0;A<t.k.length;A++)t.e[A]=new ht.IN(t.k[A]);if(t.a!=null)for(t.b=B(o.Sab,R1,105,t.a.length,0,1),A=0;A<t.a.length;A++)t.b[A]=new ht.IN(t.a[A]);t.d=24;},n.rk=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;for(P=B(o.Zab,R1,35,t.k.length,0,1),q=0,E=0,c=0;c<t.k.length;c++)n.be(t.e[c],A,null,H1),P[c]=t.e[c].v,q+=P[c].b,E=d.Math.max(E,P[c].a);for(u=B(o.Zab,R1,35,t.a.length,0,1),p=0,S=0,b=0;b<t.a.length;b++)n.be(t.b[b],A,null,65552),u[b]=t.b[b].v,p=d.Math.max(p,u[b].b),S+=u[b].a+12;if(r=d.Math.max(48,p+24),E=d.Math.max(E,24+2*S),e=-1,!t.g)t.g=new n.fk(),U.sj(t.g,new At.GH()),e=0;else {for(g=0;g<t.g.a.length;g++)if(Zt(U.xj(t.g,g),61)){e=g;break;}e==-1&&(e=t.g.a.length,U.sj(t.g,new At.GH()));}for(v=18,T=0;T<t.k.length;T++){if(T==t.n){for(At.FH(U.xj(t.g,e),v-18,E/2,v-18+r,E/2),i=v+-6,D=.5*(E-12)-S,k=0;k<t.a.length;k++)l=i+.5*(p-u[k].b)-u[k].c,y=D-u[k].d,n.dd(t.b[k],new n.Di(l,y)),D+=12+u[k].a;v+=r;}Q=v-P[T].c,L=.5*(E-P[T].a)-P[T].d,n.dd(t.e[T],new n.Di(Q,L)),v+=36+P[T].b;}t.o=false;},n.sk=function(t,A){var e,r;if(t.g)for(r=new U.rrb(t.g);r.a<r.c.a.length;)e=U.qrb(r),e.Ib(A,t.p);},n.tk=function(t,A){var e,r,P,i,D;if(t.i!=0&&t.k!=null){for(r=n.ok(t)/t.p.c,R.F3(A,t.i),R.D3(A,Mt(1.6*r),true,false),i=0;i<t.k.length;i++)if(t.k[i].q!=0){for(P=new j1.Ngb(),e=0;e<t.k[i].q;e++)P.a=Mt(P.a+n.Ql(t.k[i],e)),P.b=Mt(P.b+n.Rl(t.k[i],e));P.a=P.a/t.k[i].q|0,P.b=P.b/t.k[i].q|0,P.a=Mt(n.Ai(t.e[i].R,P.a)),P.b=Mt(n.Bi(t.e[i].R,P.b)),D=t.c==0?""+(i+1):t.c==2?i<t.j?""+String.fromCharCode(65+i&e1):"R"+(i+1-t.j):t.c==1?i<t.n?""+String.fromCharCode(65+i&e1):"P"+(i+1-t.n):"?"+(i+1),R.m3(A,P.a,P.b,D);}}},n.uk=function(t,A){var e,r,P,i,D;if(t.e!=null)for(e=n.ok(t)/t.p.c,P=t.e,i=0,D=P.length;i<D;++i)r=P[i],n.Wd(r,t.f),r.r=e/r.R.c,n.Td(r,A);if(t.b!=null)for(P=t.b,i=0,D=P.length;i<D;++i)r=P[i],n.Td(r,A);},n.vk=function(t,A){t.d=A;},n.wk=function(t,A){t.f=A;},n.xk=function(t,A){var e,r,P,i;if(t.e!=null)for(r=t.e,P=0,i=r.length;P<i;++P)e=r[P],e.J=A;},n.yk=function(t,A,e){var r,P,i,D,u,Q,l;if(t.e!=null)for(i=t.e,u=0,l=i.length;u<l;++u)r=i[u],n.Xd(r,A,e);if(t.b!=null)for(P=t.b,D=0,Q=P.length;D<Q;++D)r=P[D],n.Xd(r,A,e);},n.zk=function(t,A){t.i=A;},n.Ak=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v;if(n.Bk(t,A,e,r),n.zi(t.p))return null;if(t.k!=null)for(T=t.k,c=0,b=T.length;c<b;++c)y=T[c],n.vi(t.p,y);if(t.g)for(k=new U.rrb(t.g);k.a<k.c.a.length;)g=U.qrb(k),n.ti(t.p,g);if(t.e!=null)for(D=t.e,Q=0,L=D.length;Q<L;++Q)P=D[Q],n.yi(P.R);if(t.b!=null)for(i=t.b,u=0,l=i.length;u<l;++u)P=i[u],n.yi(P.R);return v=t.p,t.p=new n.Ci(),v;},n.Bk=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k;if(t.o&&n.rk(t,A),i=null,t.e!=null)for(u=t.e,l=0,y=u.length;l<y;++l)D=u[l],n.be(D,A,null,0),i=i?ht.nO(i,D.v):D.v;if(t.b!=null)for(u=t.b,l=0,y=u.length;l<y;++l)D=u[l],n.be(D,A,null,0),i=i?ht.nO(i,D.v):D.v;if(t.g)for(b=new U.rrb(t.g);b.a<b.c.a.length;)c=U.qrb(b),g=c.Jb(A),n.xi(t.p,g),i=i?ht.nO(i,g):g;if(!i)return null;if(P=n.ok(t),k=new n.Ei(i,e,P,r),!(k.c==1&&k.a==0&&k.b==0)){if(n.ui(k,t.p),t.e!=null)for(Q=t.e,L=0,T=Q.length;L<T;++L)D=Q[L],n.dd(D,k);if(t.b!=null)for(u=t.b,l=0,y=u.length;l<y;++l)D=u[l],n.dd(D,k);return k;}return null;},n.Ck=function(t,A){t&&(this.k=B(o.Q8,GA,26,1,0,1),this.k[0]=t),this.c=0,this.g=A,n.qk(this);},n.Dk=function(t,A){var e,r;for(this.k=B(o.Q8,GA,26,t.g.a.length+t.f.a.length,0,1),r=0;r<t.g.a.length+t.f.a.length;r++)this.k[r]=r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length);for(this.n=t.g.a.length,this.a=B(o.Q8,GA,26,t.a.a.length,0,1),e=0;e<t.a.a.length;e++)this.a[e]=U.xj(t.a,e);this.o=false,this.c=1,this.g=A,n.qk(this);},n.Ek=function(t,A,e){this.k=t,this.c=2,this.g=e,this.j=A,n.qk(this);},n.Fk=function(t,A){this.k=t,this.c=0,this.g=A,n.qk(this);},n.Gk=function(t,A){return w.UAb((w.GAb(t),t))>w.UAb((w.GAb(A),A))?-1:w.UAb((w.GAb(t),t))<w.UAb((w.GAb(A),A))?1:0;},x(158,1,{},n.Ck,n.Dk,n.Ek,n.Fk),s.c=0,s.d=0,s.f=0,s.i=0,s.j=0,s.n=0,s.o=false,o.o8=N(158),n.Hk=function(){},x(365,1,{},n.Hk),s.Rb=function(t,A){return n.Gk(t,A);},s.mb=function(t){return this===t;},o.n8=N(365),n.Rk=function(){n.Rk=f1,n.Lk=j(F(o.Ocb,1),yt,2,6,["?","H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr","Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og",M1,M1,M1,M1,M1,M1,M1,M1,M1,M1,"R4","R5","R6","R7","R8","R9","R10","R11","R12","R13","R14","R15","R16","R1","R2","R3","A","A1","A2","A3",M1,M1,"D","T","X","R","H2","H+","Nnn","HYD","Pol",M1,M1,M1,M1,M1,M1,M1,M1,M1,M1,M1,"Ala","Arg","Asn","Asp","Cys","Gln","Glu","Gly","His","Ile","Leu","Lys","Met","Phe","Pro","Ser","Thr","Trp","Tyr","Val"]),n.Pk=j(F(o.ufb,1),Yt,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]),n.Ok=j(F(o.M7,1),Y,6,15,[6]),n.Kk=j(F(o.M7,1),Y,6,15,[2]),n.Mk=j(F(o.M7,2),cA,10,0,[null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[0]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[0]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[2,4,6]),j(F(o.M7,1),Y,6,15,[1,3,5,7]),j(F(o.M7,1),Y,6,15,[0]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[2,4,6]),j(F(o.M7,1),Y,6,15,[1,3,5,7]),j(F(o.M7,1),Y,6,15,[0,2]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,j(F(o.M7,1),Y,6,15,[1,2,3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[2,4,6]),j(F(o.M7,1),Y,6,15,[1,3,5,7]),j(F(o.M7,1),Y,6,15,[0,2,4,6]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2])]),n.Nk=j(F(o.M7,2),cA,10,0,[null,j(F(o.M7,1),Y,6,15,[1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,j(F(o.M7,1),Y,6,15,[-3]),j(F(o.M7,1),Y,6,15,[-2]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),null,j(F(o.M7,1),Y,6,15,[-3]),j(F(o.M7,1),Y,6,15,[-2]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3,4]),j(F(o.M7,1),Y,6,15,[2,3,4,5]),j(F(o.M7,1),Y,6,15,[2,3,6]),j(F(o.M7,1),Y,6,15,[2,3,4,7]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[1,2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[-3,3,5]),j(F(o.M7,1),Y,6,15,[-2]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[6]),j(F(o.M7,1),Y,6,15,[4,6,7]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[-3,3,5]),j(F(o.M7,1),Y,6,15,[-2,4,6]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[5]),j(F(o.M7,1),Y,6,15,[6]),j(F(o.M7,1),Y,6,15,[4,6,7]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[1,3]),j(F(o.M7,1),Y,6,15,[1,2]),j(F(o.M7,1),Y,6,15,[1,3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[-2,2,4]),j(F(o.M7,1),Y,6,15,[-1,1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[4,5]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[3])]);},n.Sk=function(t,A){var e,r;e=-1,r=0,(t.G[A]&16)!=0&&(e=16,++r),(t.G[A]&64)!=0&&(e=64,++r),(t.G[A]&32)!=0&&(e=32,++r),(t.G[A]&4)!=0&&(e=4,++r),(t.G[A]&2)!=0&&(e=2,++r),(t.G[A]&8)!=0&&(e=8,++r),(t.G[A]&1)!=0&&(e=1,++r),e!=-1&&e!=(t.H[A]&127)&&(t.H[A]=e),r<2&&(t.G[A]&=-128);},n.Tk=function(t,A,e,r){var P;return P=n.Uk(t,6),n.ii(t.J[P],A,e,r),P;},n.Uk=function(t,A){return t.q>=t.M&&n.Rn(t,t.M*2),t.C[t.q]=0,n.Cn(t,t.q,A),t.s[t.q]=0,t.u[t.q]=0,t.B[t.q]=0,t.w[t.q]=0,n.ii(t.J[t.q],0,0,0),t.v!=null&&(t.v[t.q]=null),t.t!=null&&(t.t[t.q]=null),t.T=0,t.q++;},n.Vk=function(t,A,e){return n.Wk(t,A,e,n.Lm(t,A)||n.Lm(t,e)?16:1);},n.Wk=function(t,A,e,r){var P;if(A==e)return  -1;for(P=0;P<t.r;P++)if(t.D[0][P]==A&&t.D[1][P]==e||t.D[0][P]==e&&t.D[1][P]==A)return t.H[P]<r&&(t.H[P]=r),P;return n.Xk(t,A,e,r);},n.Xk=function(t,A,e,r){return t.r>=t.N&&n.Sn(t,t.N*2),t.D[0][t.r]=A,t.D[1][t.r]=e,t.H[t.r]=r,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},n.Yk=function(t,A){return n.Zk(t,A,A.q,A.r,true);},n.Zk=function(t,A,e,r,P){var i,D,u,Q,l;for(t.K=t.K|A.K,D=B(o.Q7,H,6,A.q,15,1),Q=n.cn(t,1),l=n.cn(t,2),i=0;i<e;i++)(P||S1(J(A.B[i],i0),0))&&(D[i]=n.ll(A,t,i,Q,l));for(u=0;u<r;u++)(P||!n.Jm(A,A.D[0][u])&&!n.Jm(A,A.D[1][u]))&&n.ml(A,t,u,Q,l,D[A.D[0][u]],D[A.D[1][u]],false);return t.L=t.L&&A.L,t.I=0,t.T=0,D;},n.$k=function(t,A,e,r,P,i,D,u){var Q,l;return Q=n.xl(t,A,e),Q==-1?(t.q>=t.M&&n.Rn(t,t.M*2),Q=n.Uk(t,r),n.ii(t.J[Q],A,e,0),t.A[Q]=P,n.gn(t,Q,i),n.wn(t,Q,D),n.mn(t,Q,u),true):(l=n.fl(t,Q,r,P,i,D),l=l|n.mn(t,Q,u),l);},n._k=function(t,A,e,r){var P;for(P=0;P<t.r;P++)if(t.D[0][P]==A&&t.D[1][P]==e||t.D[0][P]==e&&t.D[1][P]==A)return n.il(t,P,r),t.T=0,P;return t.r>=t.N&&n.Sn(t,t.N*2),t.D[0][t.r]=A,t.D[1][t.r]=e,t.H[t.r]=r,t.F[t.r]=0,t.G[t.r]=0,t.T=0,t.r++;},n.al=function(t,A,e,r,P,i){for(var D,u,Q;t.q+r>t.M;)n.Rn(t,t.M*2);for(;t.r+r>t.N;)n.Sn(t,t.N*2);return D=n.xl(t,A,e),D!=-1?n.bl(t,D,r,P,i):(u=n.yl(t,A,e),u!=-1?n.cl(t,u,r,P,i):(D=n.Tk(t,A,e,0),Q=A1*(r-2)/r,n.Wm(t,D,r,D,P,0,A1-Q,i),t.T=0,true));},n.bl=function(t,A,e,r,P){var i,D,u,Q,l,L;if(r&&n.vp(t,A)>1||!r&&n.vp(t,A)>2)return  false;for(D=0,i=B(o.O7,Ft,6,4,15,1),Q=0;Q<t.r;Q++){for(l=0;l<2;l++)if(t.D[l][Q]==A){if(D==2){D=3;break;}i[D++]=n.Yl(t,A,t.D[1-l][Q]);}if(D==3)break;}return D==3?false:(L=D==1?i[0]+A1:d.Math.abs(i[0]-i[1])>A1?(i[0]+i[1])/2:(i[0]+i[1])/2+A1,u=A1*(e-2)/e,n.Wm(t,A,e,A,r,L-u/2,A1-u,P),t.T=0,true);},n.cl=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b,g;if(l=B(o.Q7,H,6,2,15,1),Q=B(o.O7,Ft,6,2,15,1),l[0]=t.D[0][A],l[1]=t.D[1][A],n.vp(t,l[0])>3||n.vp(t,l[1])>3)return  false;for(D=0,i=B(o.O7,Ft,6,4,15,1),T=0;T<t.r;T++)if(T!=A){for(c=0;c<2;c++){for(b=0;b<2;b++)if(t.D[c][T]==l[b]){if(D==4){D=5;break;}i[D++]=n.Yl(t,l[b],t.D[1-c][T]);}if(D==5)break;}if(D==5)break;}if(D==5)return  false;for(Q[0]=n.Yl(t,l[0],l[1]),Q[0]<0?(Q[1]=Q[0]+A1,u=0):(Q[1]=Q[0],Q[0]=Q[1]-A1,u=1),g=0,y=0;y<D;y++)i[y]>Q[0]&&i[y]<Q[1]?--g:++g;return u=g>0?1-u:u,L=A1*(e-2)/e,n.Wm(t,l[u],e-1,l[1-u],r,Q[g>0?0:1]+A1-L,A1-L,P),t.T=0,true;},n.dl=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g;for(P=B(o.Q7,H,6,A.q,15,1),Q=n.cn(t,1),l=n.cn(t,2),r=0;r<A.q;r++)if(A.C[r]!=0||(A.t==null||A.t[r]==null?null:h.$jb((c=A.t[r],q1.Cnb(),c)))!=null)P[r]=n.ll(A,t,r,Q,l);else if(P[r]=e,(A.T&8)!=0){for(u=0;u<A.r;u++)for(L=0;L<2;L++)if(A.D[L][u]==r){if(b=A.D[1-L][u],g=A.u[b]&3,y=false,g==1||g==2){for(i=0;i<A.r;i++)if(i!=u)for(T=0;T<2;T++)A.D[T][i]==b&&A.D[1-T][i]<r&&(y=!y);}y&&n.un(A,b,g==1?2:1,(A.u[b]&4)!=0);}}for(D=0;D<A.r;D++)n.ml(A,t,D,Q,l,P[A.D[0][D]],P[A.D[1][D]],false);return t.L=t.L&&A.L,t.I=0,t.T=0,P;},n.el=function(t,A){var e,r,P,i,D,u,Q,l,L;return e=t.J[A[0]],r=t.J[A[1]],P=t.J[A[2]],i=t.J[A[3]],Q=new n.oi(r.a-e.a,r.b-e.b,r.c-e.c),l=new n.oi(P.a-r.a,P.b-r.b,P.c-r.c),L=new n.oi(i.a-P.a,i.b-P.b,i.c-P.c),D=new n.oi(Q.b*l.c-Q.c*l.b,-(Q.a*l.c-Q.c*l.a),Q.a*l.b-Q.b*l.a),u=new n.oi(l.b*L.c-l.c*L.b,-(l.a*L.c-l.c*L.a),l.a*L.b-l.b*L.a),-d.Math.atan2(d.Math.sqrt(l.a*l.a+l.b*l.b+l.c*l.c)*(Q.a*u.a+Q.b*u.b+Q.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},n.fl=function(t,A,e,r,P,i){var D;return (e==1||e==151||e==152)&&n.vp(t,A)>1?false:(D=t.K&&pt(J(t.B[A],-2),0),t.B[A]=J(t.B[A],-2),t.v!=null&&t.v[A]!=null&&(t.v[A]=null,D=true),t.t!=null&&t.t[A]!=null&&(t.t[A]=null,D=true),e==t.C[A]&&r==t.A[A]&&P==((t.u[A]&bA)>>>27)-1&&i==(t.u[A]&48)?D:((e==151||e==152)&&(r=e-149,e=1),t.u[A]&=960,t.C[A]=e,t.A[A]=r,t.s[A]=0,t.B[A]=0,n.gn(t,A,P),n.wn(t,A,i),n.an(t,t.w[A]),t.T=0,true));},n.gl=function(t,A,e,r){var P;return P=n.xl(t,A,e),P!=-1&&n.hl(t,P,r);},n.hl=function(t,A,e){if(e){if(t.s[A]>8)return  false;++t.s[A];}else {if(t.s[A]<-8)return  false;--t.s[A];}return t.T=0,true;},n.il=function(t,A,e){var r,P,i,D,u;return i=false,D=t.H[A],e==511?i=n.sm(t,A):n.Eq(t,A,e)&&(e==257||e==129?(r=n.Xm(t,A,t.D[0][A]),P=n.Xm(t,A,t.D[1][A]),e==D?(r==P||P)&&(u=t.D[0][A],t.D[0][A]=t.D[1][A],t.D[1][A]=u,i=true):(!r&&P&&(u=t.D[0][A],t.D[0][A]=t.D[1][A],t.D[1][A]=u),t.H[A]=e,i=true)):(t.H[A]=e,i=true)),i&&(t.T=(D&127)==(e&127)?t.T&7:0,t.G[A]=0),i;},n.jl=function(t){t.q=0,t.r=0,t.K=false,t.L=false,t.I=0,t.v=null,t.t=null,t.P=null,t.T=0;},n.kl=function(t){var A,e,r,P,i,D,u,Q;for(D=0;D<t.r;D++)t.H[D]==512&&(e=t.D[0][D],r=t.D[1][D],t.C[e]==-1^t.C[r]==-1&&t.s[e]!=0&&t.s[r]!=0&&t.s[e]<0^t.s[r]<0&&(t.s[e]<0?(++t.s[e],--t.s[r]):(--t.s[e],++t.s[r])));for(Q=B(o.Q7,H,6,t.q,15,1),P=0,A=0;A<t.q;A++){if(t.C[A]==-1){Q[A]=-1;continue;}P<A&&(t.C[P]=t.C[A],t.s[P]=t.s[A],t.A[P]=t.A[A],t.u[P]=t.u[A],t.B[P]=t.B[A],t.w[P]=t.w[A],n.ji(t.J[P],t.J[A]),t.v!=null&&(t.v[P]=t.v[A]),t.t!=null&&(t.t[P]=t.t[A])),Q[A]=P,++P;}for(t.q=P,u=0,i=0;i<t.r;i++)t.H[i]!=512&&(t.H[u]=t.H[i],t.F[u]=t.F[i],t.G[u]=t.G[i],t.D[0][u]=Q[t.D[0][i]],t.D[1][u]=Q[t.D[1][i]],++u);return t.r=u,Q;},n.ll=function(t,A,e,r,P){var i,D,u;return i=A.q,i>=A.M&&n.Rn(A,A.M*2),u=(t.u[e]&l1)>>18,D=-1,u==1?r==-1?D=n.cn(A,u):D=d.Math.min(31,r+((t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&F0)>>20)):u==2&&(P==-1?D=n.cn(A,u):D=d.Math.min(31,P+((t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&F0)>>20))),A.C[i]=t.C[e],A.s[i]=t.s[e],A.A[i]=t.A[e],A.u[i]=t.u[e],A.B[i]=A.K?t.B[e]:0,n.ji(A.J[i],t.J[e]),A.w[i]=t.w[e],A.v!=null&&(A.v[i]=null),t.v!=null&&t.v[e]!=null&&A.K&&(A.v==null&&(A.v=B(o.Q7,E1,7,A.C.length,0,2)),A.v[i]=U.zrb(t.v[e],t.v[e].length)),A.t!=null&&(A.t[i]=null),t.t!=null&&t.t[e]!=null&&(A.t==null&&(A.t=B(o.M7,cA,10,A.C.length,0,2)),A.t[i]=U.xrb(t.t[e],t.t[e].length)),D!=-1&&(A.u[i]&=-32505857,A.u[i]|=D<<20),++A.q,A.T=0,i;},n.ml=function(t,A,e,r,P,i,D,u){var Q,l,L,y;return l=A.r,l>=A.N&&n.Sn(A,A.N*2),y=(t.F[e]&768)>>8,L=-1,y==1&&(r==-1?L=n.cn(A,y):L=d.Math.min(32,r+((t.F[e]&768)>>8!=1&&(t.F[e]&768)>>8!=2?-1:(t.F[e]&E0)>>10))),y==2&&(P==-1?L=n.cn(A,y):L=d.Math.min(32,P+((t.F[e]&768)>>8!=1&&(t.F[e]&768)>>8!=2?-1:(t.F[e]&E0)>>10))),A.D[0][l]=i,A.D[1][l]=D,Q=u&&t.Zb(e)?8:t.H[e],A.H[l]=Q,A.F[l]=t.F[e],A.G[l]=A.K?t.G[e]:0,L!=-1&&(A.F[l]&=-31745,A.F[l]|=L<<10),++A.r,A.T=0,l;},n.nl=function(t,A,e,r,P,i,D){return n.ml(t,A,e,r,P,i==null?t.D[0][e]:i[t.D[0][e]],i==null?t.D[1][e]:i[t.D[1][e]],D);},n.ol=function(t,A){var e,r;for(A.v=null,A.t=null,A.K=t.K,A.q=0,e=0;e<t.q;e++)n.ll(t,A,e,0,0);for(A.r=0,r=0;r<t.r;r++)n.ml(t,A,r,0,0,t.D[0][r],t.D[1][r],false);t.Yb(A);},n.pl=function(t,A){A.K=t.K,A.L=t.L,A.S=t.S,A.I=t.I,A.P=t.P,A.T=t.T&24;},n.ql=function(t,A){var e,r;for(e=0;e<t.r;e++)for(r=0;r<2;r++)t.D[r][e]==A&&(t.H[e]=512);n.an(t,t.w[A]),t.C[A]=-1,t.v!=null&&(t.v[A]=null),t.t!=null&&(t.t[A]=null),n.kl(t),t.T=0;},n.rl=function(t,A,e){var r,P;return r=n.xl(t,A,e),r!=-1?((t.u[r]&512)!=0?n.wl(t):n.ql(t,r),t.T=0,true):(P=n.yl(t,A,e),P!=-1?((t.u[t.D[0][P]]&t.u[t.D[1][P]]&512)!=0?n.wl(t):n.ul(t,P),t.T=0,true):false);},n.sl=function(t,A){var e,r,P,i;if(A.length==0)return null;for(r=A,P=0,i=r.length;P<i;++P)e=r[P],t.C[e]=-1;return n.vl(t);},n.tl=function(t,A){t.H[A]=512,n.kl(t),t.T=0;},n.ul=function(t,A){var e,r,P;for(r=0;r<2;r++){for(e=0,P=0;P<t.r;P++)P!=A&&(t.D[0][P]==t.D[r][A]||t.D[1][P]==t.D[r][A])&&++e;e==0&&(n.an(t,t.w[t.D[r][A]]),t.C[t.D[r][A]]=-1);}t.H[A]=512,n.kl(t),t.T=0;},n.vl=function(t){var A,e,r;for(r=false,A=0;A<t.q;A++)t.C[A]==-1&&(r=true,n.an(t,t.w[A]));for(e=0;e<t.r;e++)t.H[e]==512?r=true:(t.C[t.D[0][e]]==-1||t.C[t.D[1][e]]==-1)&&(t.H[e]=512,r=true);return r?(t.T=0,n.kl(t)):null;},n.wl=function(t){var A,e;for(e=false,A=0;A<t.q;A++)(t.u[A]&512)!=0&&(t.C[A]=-1,e=true);return e&&n.vl(t)!=null;},n.xl=function(t,A,e){var r,P,i,D,u,Q,l,L;for(D=-1,P=n.Wl(t,t.q,t.r,n.Qk),u=s0,Q=P*P/12,r=0;r<t.q;r++)l=t.J[r].a,L=t.J[r].b,i=(A-l)*(A-l)+(e-L)*(e-L),i<Q&&i<u&&(u=i,D=r);return D;},n.yl=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;for(T=-1,b=n.Wl(t,t.q,t.r,n.Qk),c=s0,r=0;r<t.r;r++)g=t.J[t.D[0][r]].a,v=t.J[t.D[0][r]].b,k=t.J[t.D[1][r]].a,S=t.J[t.D[1][r]].b,L=k-g,y=S-v,P=d.Math.sqrt(L*L+y*y),i=(g+k)/2,D=(v+S)/2,L=A-i,y=e-D,!(d.Math.sqrt(L*L+y*y)>P/2)&&(k==g?l=d.Math.abs(g-A):(u=(S-v)/(g-k),Q=-u*g-v,l=d.Math.abs((u*A+e+Q)/d.Math.sqrt(u*u+1))),l<b&&l<c&&(c=l,T=r));return T;},n.zl=function(t,A){return ((t.u[A]&bA)>>>27)-1;},n.Al=function(t,A){return (t.u[A]&49152)>>14;},n.Bl=function(t,A){return t.s[A];},n.Cl=function(t,A){return t.u[A]&448;},n.Dl=function(t,A){return t.J[A];},n.El=function(t,A){var e;return t.t==null||t.t[A]==null?null:h.$jb((e=t.t[A],q1.Cnb(),e));},n.Fl=function(t,A){return t.t==null?null:t.t[A];},n.Gl=function(t,A){return (t.u[A]&l1)>>18!=1&&(t.u[A]&l1)>>18!=2?-1:(t.u[A]&F0)>>20;},n.Hl=function(t,A){return (t.u[A]&l1)>>18;},n.Il=function(t,A){return n.Lk[t.C[A]];},n.Jl=function(t,A){return t.v==null?null:t.v[A];},n.Kl=function(t,A){var e,r,P;if(t.v==null||t.v[A]==null)return pt(J(t.B[A],1),0)?"":n.Lk[t.C[A]];for(P="",r=0;r<t.v[A].length;r++)r>0&&(P=(w.GAb(P),P+(w.GAb(","),","))),e=t.v[A][r],P=h.Yjb(P,n.Lk[e]);return P;},n.Ll=function(t,A){return d.Math.abs(t.w[A]);},n.Ml=function(t,A){return t.A[A];},n.Nl=function(t,A){return t.u[A]&3;},n.Ol=function(t,A){return t.B[A];},n.Pl=function(t,A){return t.u[A]&48;},n.Ql=function(t,A){return t.J[A].a;},n.Rl=function(t,A){return t.J[A].b;},n.Sl=function(t,A){return t.J[A].c;},n.Tl=function(t,A){return t.C[A];},n.Ul=function(t){return n.Wl(t,t.q,t.r,n.Qk);},n.Vl=function(t,A){return n.Wl(t,t.q,t.r,A);},n.Wl=function(t,A,e,r){return n.Xl(t,A,e,r,t.J);},n.Xl=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b;for(y=false,T=0,l=0;l<e;l++)t.H[l]!=16&&(t.G[l]&sA)==0&&++T;if(T==0){for(L=0;L<e;L++)(t.G[L]&sA)==0&&++T;y=true;}if(T==0){if(A<2)return r;for(b=s0,i=1;i<A;i++)for(D=0;D<i;D++)c=n.di(P[i],P[D]),c>0&&c<b&&(b=c);return b!=s0?.6*b:r;}for(u=0,Q=0;Q<e;Q++)(y||t.H[Q]!=16)&&(t.G[Q]&sA)==0&&(u+=n.di(P[t.D[1][Q]],P[t.D[0][Q]]));return u/T;},n.Yl=function(t,A,e){return n.no(t.J[A].a,t.J[A].b,t.J[e].a,t.J[e].b);},n.Zl=function(t,A,e){return t.D[A][e];},n.$l=function(t,A){return ((t.G[A]&7680)>>9)+((t.G[A]&$e)>>13);},n._l=function(t,A){return (t.G[A]&7680)>>9;},n.am=function(t,A){return (t.F[A]&48)>>4;},n.bm=function(t,A){return (t.F[A]&768)>>8!=1&&(t.F[A]&768)>>8!=2?-1:(t.F[A]&E0)>>10;},n.cm=function(t,A){return (t.F[A]&768)>>8;},n.dm=function(t,A){var e,r,P,i;return e=t.D[0][A],r=t.D[1][A],P=t.J[r].a-t.J[e].a,i=t.J[r].b-t.J[e].b,d.Math.sqrt(P*P+i*i);},n.em=function(t,A,e){var r;for(r=0;r<t.r;r++)if((t.D[0][r]==A&&t.D[1][r]==e||t.D[0][r]==e&&t.D[1][r]==A)&&t.H[r]!=512)return r;return  -1;},n.fm=function(t,A){if(t.K&&(t.G[A]&127)!=0){if((t.G[A]&9)!=0)return 1;if((t.G[A]&2)!=0)return 2;if((t.G[A]&4)!=0)return 3;if((t.G[A]&32)!=0)return 4;if((t.G[A]&64)!=0)return 5;if((t.G[A]&16)!=0)return 0;}switch(t.H[A]&127){case 1:case 8:return 1;case 2:return 2;case 4:return 3;case 32:return 4;case 64:return 5;default:return 0;}},n.gm=function(t,A){return t.F[A]&3;},n.hm=function(t,A){return t.G[A];},n.im=function(t,A){return t.H[A];},n.jm=function(t,A){return t.H[A]&127;},n.km=function(t,A){var e,r,P,i,D;if(t.q==0)return null;for(r=t.J[0].a,i=t.J[0].b,P=t.J[0].a,D=t.J[0].b,e=1;e<t.q;e++)r>t.J[e].a?r=t.J[e].a:P<t.J[e].a&&(P=t.J[e].a),i>t.J[e].b?i=t.J[e].b:D<t.J[e].b&&(D=t.J[e].b);return A?(A.c=r,A.d=i,A.b=P-r,A.a=D-i):A=new ht.pO(r,i,P-r,D-i),A;},n.lm=function(t,A){var e;return e=t.C[A]<n.Mk.length?n.Mk[t.C[A]]:null,e==null?6:e[e.length-1];},n.mm=function(t,A,e){return n.nm(t,A,e,t.C[A]);},n.nm=function(t,A,e,r){var P,i;return r>=171&&r<=190?0:(i=0,(t.u[A]&48)==32&&(i-=1),((t.u[A]&48)==16||(t.u[A]&48)==48)&&(i-=2),P=t.s[A],P==0&&t.K&&(S1(J(t.B[A],mA),on)&&(P=-1),S1(J(t.B[A],mA),Qn)&&(P=1)),r==7||r==8||r==9?i+=P:r==6||r==14||r==32?i-=d.Math.abs(P):r==15||r==33?e-i-P<=3?i+=P:i-=P:r==16||r==34||r==52?e-i-P<=4?i+=P:i-=d.Math.abs(P):r==17||r==35||r==53?e-i-P<=5?i+=P:i-=d.Math.abs(P):i-=P,i);},n.om=function(t,A){var e;return e=n.pm(t,A),e+n.nm(t,A,e,t.C[A]);},n.pm=function(t,A){var e,r;return e=((t.u[A]&bA)>>>27)-1,e==-1&&(e=(r=t.C[A]<n.Mk.length?n.Mk[t.C[A]]:null,r==null?6:r[r.length-1])),e;},n.qm=function(t,A){var e,r,P,i,D;for(i=n.Tm(t,t.D[0][A])||n.Tm(t,t.D[1][A])?5:3,r=0;r<2;r++)e=t.D[r][A],P=n.fm(t,A)+(D=n.pm(t,e),D+n.nm(t,e,D,t.C[e]))-n.vp(t,e),i>P&&(i=P);return i;},n.rm=function(t,A){return (t.u[A]&H1)!=0;},n.sm=function(t,A){var e,r,P;return r=n.qm(t,A),e=n.Lm(t,t.D[0][A])||n.Lm(t,t.D[1][A]),P=e?16:1,t.H[A]==64?(t.H[A]=P,t.T=0,true):t.H[A]==32?(t.H[A]=r>4?64:P,t.T=0,true):t.H[A]==4?(t.H[A]=r>3?32:P,t.T=0,true):t.H[A]==2&&(t.H[A]=386,t.T&=7,(t.F[A]&128)==0)?true:t.H[A]==386?(r>2?t.H[A]=4:t.H[A]=P,t.T=0,true):(384&t.H[A])!=0?(t.H[A]=1,t.T&=7,true):!e&&r<2?false:t.H[A]==1?(t.H[A]=2,t.T=0,true):r<1?false:t.H[A]==16?(t.H[A]=1,t.T=0,true):false;},n.tm=function(t){var A;for(t.T=0,t.C=B(o.Q7,H,6,t.M,15,1),t.s=B(o.Q7,H,6,t.M,15,1),t.w=B(o.Q7,H,6,t.M,15,1),t.J=B(o.h8,En,29,t.M,0,1),A=0;A<t.M;A++)t.J[A]=new n.ni();t.A=B(o.Q7,H,6,t.M,15,1),t.u=B(o.Q7,H,6,t.M,15,1),t.B=B(o.R7,e0,6,t.M,14,1),t.v=null,t.t=null,t.D=U1(o.Q7,[E1,H],[7,6],15,[2,t.N],2),t.H=B(o.Q7,H,6,t.N,15,1),t.F=B(o.Q7,H,6,t.N,15,1),t.G=B(o.Q7,H,6,t.N,15,1);},n.um=function(t,A){t.T&=~A;},n.vm=function(t){var A;for(A=0;A<t.q;A++)if(t.J[A].c!=0)return  true;return  false;},n.wm=function(t,A){return (t.u[A]&oe)!=0;},n.xm=function(t,A){return t.C[A]==-1;},n.ym=function(t,A){return (t.u[A]&4)!=0;},n.zm=function(t,A){return (t.u[A]&q0)!=0;},n.Am=function(t,A){return t.w[A]<0;},n.Bm=function(t,A){return (t.F[A]&LA)!=0;},n.Cm=function(t,A){return (t.G[A]&sA)!=0;},n.Dm=function(t,A){return (t.F[A]&H1)!=0;},n.Em=function(t,A){return t.H[A]==512;},n.Fm=function(t,A){return (t.F[A]&4)!=0;},n.Gm=function(t,A){return (t.F[A]&r0)!=0;},n.Hm=function(t,A){var e,r,P,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!n.qo(e))return  false;}}return n.qo(t.C[A]);},n.Im=function(t,A){var e,r,P,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!n.ro(e))return  false;}}return n.ro(t.C[A]);},n.Jm=function(t,A){return pt(J(t.B[A],i0),0);},n.Km=function(t,A){return (t.u[A]&r0)!=0;},n.Lm=function(t,A){var e,r,P,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!(e>=3&&e<=4||e>=11&&e<=13||e>=19&&e<=31||e>=37&&e<=51||e>=55&&e<=84||e>=87&&e<=112))return  false;}}return n.so(t.C[A]);},n.Mm=function(t,A){return t.A[A]==0;},n.Nm=function(t,A){var e,r,P,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!(e==1||e>=5&&e<=9||e>=14&&e<=17||e>=32&&e<=35||e>=52&&e<=53))return  false;}}return n.uo(t.C[A]);},n.Om=function(t){var A;for(A=0;A<t.q;A++)switch(t.C[A]){case 1:case 5:case 6:case 7:case 8:case 9:case 14:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:continue;default:return  false;}return  true;},n.Pm=function(t,A){return (t.u[A]&512)!=0;},n.Qm=function(t,A){return (t.u[t.D[0][A]]&t.u[t.D[1][A]]&512)!=0;},n.Rm=function(t,A){return t.H[A]==257||t.H[A]==129;},n.Sm=function(t,A,e){return (t.H[A]==257||t.H[A]==129)&&t.D[0][A]==e;},n.Tm=function(t,A){var e,r,P,i;if(t.K){if(pt(J(t.B[A],1),0))return  false;if(t.v!=null&&t.v[A]!=null){for(r=t.v[A],P=0,i=r.length;P<i;++P)if(e=r[P],!(e>=21&&e<=30||e>=39&&e<=48||e==57||e>=72&&e<=80||e==89||e>=104&&e<=112))return  false;}}return n.vo(t.C[A]);},n.Um=function(t,A){t.C[A]=-1;},n.Vm=function(t,A){t.H[A]=512;},n.Wm=function(t,A,e,r,P,i,D,u){var Q,l,L,y,T,c,b,g,k,v;for(A!=r&&(k=t.J[A].a-t.J[r].a,v=t.J[A].b-t.J[r].b,u=d.Math.sqrt(k*k+v*v)),Q=A,L=n.Wn(t,A)!=3,g=1;g<e;g++){for(T=t.J[Q].a+u*d.Math.sin(i),c=t.J[Q].b+u*d.Math.cos(i),b=-1,y=0;y<t.q;y++)if(d.Math.abs(T-t.J[y].a)<4&&d.Math.abs(c-t.J[y].b)<4){b=y;break;}b==-1&&(b=n.Tk(t,T,c,0),t.J[b].a=T,t.J[b].b=c,t.J[b].c=0),l=n.em(t,Q,b),l==-1&&(l=n.Wk(t,Q,b,n.Lm(t,Q)||n.Lm(t,b)?16:1),P&&(L&&n.Wn(t,t.D[0][l])<4&&n.Wn(t,t.D[1][l])<3&&(t.H[l]=2),L=!L)),Q=b,i+=D;}l=n.em(t,Q,r),l==-1&&(l=n.Wk(t,Q,r,n.Lm(t,Q)||n.Lm(t,r)?16:1)),P&&L&&n.Wn(t,t.D[0][l])<4&&n.Wn(t,t.D[1][l])<4&&(t.H[l]=2);},n.Xm=function(t,A,e){var r,P;if(n.fm(t,A)!=1)return  false;if((t.u[e]&3)!=0)return  true;for(P=0;P<t.r;P++)if(P!=A&&t.H[P]==2&&(t.D[0][P]==e&&(t.u[t.D[1][P]]&3)!=0||t.D[1][P]==e&&(t.u[t.D[0][P]]&3)!=0))return  true;for(r=0;r<t.r;r++)if(r!=A&&t.H[r]==1&&(t.D[0][r]==e||t.D[1][r]==e)&&(t.F[r]&3)!=0)return  true;return  false;},n.Ym=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=-449;},n.Zm=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=ae;},n.$m=function(t){var A;for(A=0;A<t.q;A++)t.u[A]&=-513;},n._m=function(t){var A;for(A=0;A<t.r;A++)t.F[A]&=-98305;},n.an=function(t,A){var e;for(e=0;e<t.q;e++)d.Math.abs(t.w[e])==d.Math.abs(A)&&(t.w[e]=0);},n.bn=function(t){var A,e,r,P;for(P=false,e=0;e<t.q;e++)pt(J(t.B[e],i0),0)&&(t.C[e]=-1,P=true);for(P&&n.vl(t),t.v!=null&&(t.v=null,P=true),A=0;A<t.q;A++)pt(t.B[A],0)&&(t.B[A]=0,P=true);for(r=0;r<t.r;r++)t.G[r]!=0&&(t.G[r]=0,P=true),t.H[r]==8&&(t.H[r]=1,P=true);return P&&(t.T=0),P;},n.cn=function(t,A){var e,r,P,i,D,u,Q,l,L;if(A==0)return 0;for(u=null,r=0;r<t.q;r++)(t.u[r]&l1)>>18==A&&(u==null&&(u=B(o.vfb,Tt,6,32,16,1)),u[(t.u[r]&l1)>>18!=1&&(t.u[r]&l1)>>18!=2?-1:(t.u[r]&F0)>>20]=true);for(i=0;i<t.r;i++)(t.F[i]&768)>>8==A&&(u==null&&(u=B(o.vfb,Tt,6,32,16,1)),u[(t.F[i]&768)>>8!=1&&(t.F[i]&768)>>8!=2?-1:(t.F[i]&E0)>>10]=true);if(L=0,u!=null){for(l=B(o.Q7,H,6,32,15,1),Q=0;Q<32;Q++)u[Q]&&(l[Q]=L++);for(e=0;e<t.q;e++)(t.u[e]&l1)>>18==A&&(D=l[(t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&F0)>>20],t.u[e]&=-32505857,t.u[e]|=D<<20);for(P=0;P<t.r;P++)(t.F[P]&768)>>8==A&&(D=l[(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10],t.F[P]&=-31745,t.F[P]|=D<<10);}return L;},n.dn=function(t,A){var e;for(e=0;e<t.q;e++)t.J[e].a*=A,t.J[e].b*=A;},n.en=function(t,A){t.q=A,t.T=0;},n.fn=function(t,A){t.r=A,t.T=0;},n.gn=function(t,A,e){e>=-1&&e<=14&&(t.u[A]&=-2013265921,t.u[A]|=1+e<<27,t.C[A]==6&&(e==-1||e==0||e==2||e==4)&&(t.u[A]&=-49,e==2&&(t.u[A]|=16)));},n.hn=function(t,A,e){t.u[A]&=-49153,t.u[A]|=e<<14;},n.jn=function(t,A,e){t.s[A]=e,t.T=0;},n.kn=function(t,A,e){t.u[A]&=-449,t.u[A]|=e;},n.ln=function(t,A,e){e?t.u[A]|=oe:t.u[A]&=tr,t.T&=7;},n.mn=function(t,A,e){var r,P,i;if(e!=null&&(w.SAb(e).length==0?e=null:(r=n.po(e,321),(r!=0&&r==t.C[A]||h.akb(e,"?"))&&(n.Cn(t,A,r),e=null))),e==null){if(t.t!=null&&t.t[A]!=null)return t.t[A]=null,true;}else if(t.t==null&&(t.t=B(o.M7,cA,10,t.M,0,2)),t.t[A]==null||!h.akb(h.$jb((i=t.t[A],q1.Cnb(),i)),e))return t.t[A]=w.sAb((P=e,q1.Cnb(),P)),true;return  false;},n.nn=function(t,A,e){e!=null&&e.length==0&&(e=null),e==null?t.t!=null&&(t.t[A]=null):(t.t==null&&(t.t=B(o.M7,cA,10,t.M,0,2)),t.t[A]=e);},n.on=function(t,A,e,r){var P,i,D;if(e==0)t.u[A]&=Ar,t.u[A]|=e<<18;else {if(r>=32)return;if(r==-1){for(D=-1,i=0;i<t.q;i++)i!=A&&e==(t.u[i]&l1)>>18&&D<((t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&F0)>>20)&&(D=(t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&F0)>>20);for(P=0;P<t.r;P++)e==(t.F[P]&768)>>8&&D<((t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10)&&(D=(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10);if(r=D+1,r>=32)return;}t.u[A]&=Ar,t.u[A]|=e<<18|r<<20;}t.T&=7;},n.pn=function(t,A,e){t.v==null&&(t.v=B(o.Q7,E1,7,t.M,0,2)),w.lAb(e,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),t.v[A]=e,t.T=0,t.K=true;},n.qn=function(t,A,e,r){var P;if(e==null){t.v!=null&&(t.v[A]=null);return;}if(e.length==1&&!r){P=e[0],t.C[A]!=P&&n.fl(t,A,P,0,-1,0),t.v!=null&&(t.v[A]=null);return;}t.v==null&&(t.v=B(o.Q7,E1,7,t.M,0,2)),t.v[A]=e,r&&(t.B[A]=Pt(t.B[A],1)),t.T=0,t.K=true;},n.rn=function(t,A,e,r){t.w[A]=r?-e:e;},n.sn=function(t,A,e){e?t.u[A]|=r0:t.u[A]&=ae;},n.tn=function(t,A,e){t.A[A]=e,t.T&=t.C[A]==1?0:7;},n.un=function(t,A,e,r){t.u[A]&=-8,e!=3&&(t.u[A]&=tr),t.u[A]|=e,r&&(t.u[A]|=4);},n.vn=function(t,A,e,r){r?t.B[A]=Pt(t.B[A],e):t.B[A]=J(t.B[A],tA(e)),t.T=0,t.K=true;},n.wn=function(t,A,e){t.u[A]&=-49,t.u[A]|=e,t.T&=7;},n.xn=function(t,A,e){e?t.u[A]|=512:t.u[A]&=-513;},n.yn=function(t,A,e){t.u[A]&=-67108865,e&&(t.u[A]|=q0);},n.zn=function(t,A,e){t.J[A].a=e,t.T&=7;},n.An=function(t,A,e){t.J[A].b=e,t.T&=7;},n.Bn=function(t,A,e){t.J[A].c=e,t.T&=7;},n.Cn=function(t,A,e){e>=0&&e<=190&&(e==151||e==152?(t.C[A]=1,t.A[A]=e-149):(t.C[A]=e,t.A[A]=0),t.u[A]&=-2013265921,t.T=0);},n.Dn=function(t,A,e,r){t.D[A][e]=r,t.T=0;},n.En=function(t,A,e){e?t.F[A]|=LA:t.F[A]&=-32769;},n.Fn=function(t,A,e){t.F[A]&=-49,t.F[A]|=e<<4;},n.Gn=function(t,A,e,r){var P,i,D;if(e==0)t.F[A]&=-32513,t.F[A]|=e<<8;else {if(r>=32)return;if(r==-1){for(D=-1,i=0;i<t.q;i++)e==(t.u[i]&l1)>>18&&D<((t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&F0)>>20)&&(D=(t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&F0)>>20);for(P=0;P<t.r;P++)P!=A&&e==(t.F[P]&768)>>8&&D<((t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10)&&(D=(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10);if(r=D+1,r>=32)return;}t.F[A]&=-32513,t.F[A]|=e<<8|r<<10;}t.T&=7;},n.Hn=function(t,A,e){e?t.F[A]|=H1:t.F[A]&=-65537;},n.In=function(t,A,e){t.H[A]=e==1?1:e==2?2:e==3?4:16,t.T=0;},n.Jn=function(t,A,e,r){t.F[A]&=-131080,t.F[A]|=e,r&&(t.F[A]|=4);},n.Kn=function(t,A){t.F[A]|=r0;},n.Ln=function(t,A,e,r){r?t.G[A]|=e:t.G[A]&=~e,t.T=0,r&&e!=0&&(t.K=true);},n.Mn=function(t,A,e){t.H[A]=e,t.T=0;},n.Nn=function(t,A){t.I=A;},n.On=function(t,A){t.K!=A&&(t.K=A,A||n.bn(t),t.T=0);},n.Pn=function(t,A){t.T=A;},n.Qn=function(t,A){t.S=A;},n.Rn=function(t,A){var e,r;for(t.C=n.ho(t.C,A),t.s=n.ho(t.s,A),t.w=n.ho(t.w,A),r=t.J.length,t.J=n.jo(t.J,A),e=r;e<A;e++)t.J[e]=new n.ni();t.A=n.ho(t.A,A),t.u=n.ho(t.u,A),t.B=n.io(t.B,A),t.v!=null&&(t.v=n.lo(t.v,A)),t.t!=null&&(t.t=n.ko(t.t,A)),t.M=A;},n.Sn=function(t,A){t.D[0]=n.ho(t.D[0],A),t.D[1]=n.ho(t.D[1],A),t.H=n.ho(t.H,A),t.F=n.ho(t.F,A),t.G=n.ho(t.G,A),t.N=A;},n.Tn=function(t,A){t.O=A;},n.Un=function(t,A){t.P=A;},n.Vn=function(t,A){t.u[A]|=H1;},n.Wn=function(t,A){var e,r;for(r=0,e=0;e<t.r;e++)(t.D[0][e]==A||t.D[1][e]==A)&&(r+=n.fm(t,e));return r;},n.Xn=function(t){var A,e,r;for(e=false,r=false,A=0;A<t.q;A++)t.A[A]!=0&&(t.A[A]=0,e=true,t.C[A]==1&&(r=true));return r&&(t.T=0),e;},n.Yn=function(t,A,e){return n.Lm(t,A)||n.Lm(t,e)?16:1;},n.Zn=function(t,A,e){var r,P,i,D,u,Q;for(D=t.C[A],t.C[A]=t.C[e],t.C[e]=D,D=t.s[A],t.s[A]=t.s[e],t.s[e]=D,D=t.A[A],t.A[A]=t.A[e],t.A[e]=D,D=t.u[A],t.u[A]=t.u[e],t.u[e]=D,Q=t.B[A],t.B[A]=t.B[e],t.B[e]=Q,D=t.w[A],t.w[A]=t.w[e],t.w[e]=D,i=t.J[A],t.J[A]=t.J[e],t.J[e]=i,t.v!=null&&(u=t.v[A],t.v[A]=t.v[e],t.v[e]=u),t.t!=null&&(u=t.t[A],t.t[A]=t.t[e],t.t[e]=u),r=0;r<t.r;r++)for(P=0;P<2;P++)t.D[P][r]==A?t.D[P][r]=e:t.D[P][r]==e&&(t.D[P][r]=A);t.T=0;},n.$n=function(t,A,e){var r;r=t.D[0][A],t.D[0][A]=t.D[0][e],t.D[0][e]=r,r=t.D[1][A],t.D[1][A]=t.D[1][e],t.D[1][e]=r,r=t.H[A],t.H[A]=t.H[e],t.H[e]=r,r=t.F[A],t.F[A]=t.F[e],t.F[e]=r,r=t.G[A],t.G[A]=t.G[e],t.G[e]=r,t.T=0;},n._n=function(t,A,e){var r;for(r=0;r<t.q;r++)t.J[r].a+=A,t.J[r].b+=e;t.U+=A,t.V+=e;},n.ao=function(t,A,e){var r,P;switch(P=e&127,r=n.qm(t,A),P){case 1:case 8:return r>=1;case 2:return r>=2;case 4:return r>=3;case 32:return r>=4;case 64:return r>=5;case 16:return  true;default:return  false;}},n.bo=function(t,A,e,r){var P,i,D;for(P=0;P<t.q;P++)(!r||(t.u[P]&512)!=0)&&(D=t.R[P]*A,i=t.Q[P]-e,t.J[P].a=t.U+D*d.Math.sin(i),t.J[P].b=t.V+D*d.Math.cos(i));r&&(t.T&=7);},n.co=function(t,A,e){var r,P,i;for(t.U=A,t.V=e,t.Q=B(o.O7,Ft,6,t.q,15,1),t.R=B(o.O7,Ft,6,t.q,15,1),r=0;r<t.q;r++)P=A-t.J[r].a,i=e-t.J[r].b,t.R[r]=d.Math.sqrt(P*P+i*i),t.Q[r]=n.no(A,e,t.J[r].a,t.J[r].b);},n.eo=function(){n.Rk(),this.M=this.N=256,n.tm(this);},n.fo=function(t,A){this.M=d.Math.max(1,t),this.N=d.Math.max(1,A),n.tm(this);},n.ho=function(t,A){var e;return e=B(o.Q7,H,6,A,15,1),h.Qkb(t,0,e,0,d.Math.min(t.length,A)),e;},n.io=function(t,A){var e;return e=B(o.R7,e0,6,A,14,1),h.Qkb(t,0,e,0,d.Math.min(t.length,A)),e;},n.jo=function(t,A){var e,r;for(e=B(o.h8,En,29,A,0,1),r=0;r<t.length;r++)t[r]&&(e[r]=new n.pi(t[r]));return e;},n.ko=function(t,A){var e,r;for(e=B(o.M7,cA,10,A,0,2),r=0;r<t.length;r++)t[r]!=null&&(e[r]=B(o.M7,Y,6,t[r].length,15,1),h.Qkb(t[r],0,e[r],0,t[r].length));return e;},n.lo=function(t,A){var e,r;for(e=B(o.Q7,E1,7,A,0,2),r=0;r<t.length;r++)t[r]!=null&&(e[r]=B(o.Q7,H,6,t[r].length,15,1),h.Qkb(t[r],0,e[r],0,t[r].length));return e;},n.mo=function(t){return n.Rk(),t>=0&&t<n.Mk.length&&n.Mk[t]!=null?n.Mk[t]:t>=171&&t<=190?n.Kk:n.Ok;},n.no=function(t,A,e,r){n.Rk();var P,i,D;return i=e-t,D=r-A,D!=0?(P=d.Math.atan(i/D),D<0&&(i<0?P-=A1:P+=A1)):P=i>0?v0:ue,P;},n.oo=function(t,A){n.Rk();var e;for(e=t-A;e<oP;)e+=N1;for(;e>A1;)e-=N1;return e;},n.po=function(t,A){n.Rk();var e,r,P,i,D;if((A&256)!=0&&h.akb(t,"?"))return 0;for(r=1;r<=128;r++)if(!h.akb(t,M1)&&h.bkb(t,n.Lk[r]))return r;if((A&2)!=0){for(P=129;P<=144;P++)if(h.bkb(t,n.Lk[P]))return P;}if((A&4)!=0){for(i=146;i<=148;i++)if(h.bkb(t,n.Lk[i]))return i;}if((A&1)!=0){for(D=151;D<=152;D++)if(h.bkb(t,n.Lk[D]))return D;}if((A&32)!=0&&h.bkb(t,n.Lk[153]))return 153;if((A&8)!=0&&h.bkb(t,n.Lk[154]))return 154;if((A&16)!=0&&h.bkb(t,n.Lk[145]))return 145;if((A&128)!=0&&h.bkb(t,n.Lk[159]))return 159;if((A&64)!=0){for(e=171;e<=190;e++)if(h.bkb(t,n.Lk[e]))return e;}return 0;},n.qo=function(t){switch(n.Rk(),t){case 7:case 8:case 9:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:return  true;}return  false;},n.ro=function(t){return n.Rk(),!(t==1||t==6||n.qo(t)||t==2||t==10||t==18||t==36||t==54||t==86||t>103);},n.so=function(t){return t>=3&&t<=4||t>=11&&t<=13||t>=19&&t<=31||t>=37&&t<=51||t>=55&&t<=84||t>=87&&t<=112;},n.to=function(t){return t==2||t==10||t==18||t==36||t==54||t==86;},n.uo=function(t){return t==1||t>=5&&t<=9||t>=14&&t<=17||t>=32&&t<=35||t>=52&&t<=53;},n.vo=function(t){return t>=21&&t<=30||t>=39&&t<=48||t==57||t>=72&&t<=80||t==89||t>=104&&t<=112;},x(88,1,{88:1,4:1},n.eo),s.Yb=function(t){n.pl(this,t);},s.Zb=function(t){return this.H[t]==8;},s.q=0,s.r=0,s.I=0,s.K=false,s.L=false,s.M=0,s.N=0,s.O=0,s.S=false,s.T=0,s.U=0,s.V=0,n.Qk=24,o.w8=N(88),n.xo=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T;for(n.bw(A,1),r==null&&(r=B(o.Q7,H,6,A.q,15,1)),u=n.cn(t,1),Q=n.cn(t,2),T=B(o.vfb,Tt,6,A.q,16,1),l=B(o.Q7,H,6,A.q,15,1),l[0]=e,T[e]=true,r[e]=n.ll(A,t,e,u,Q),D=0,L=0;D<=L;){for(y=0;y<n.Lo(A,l[D]);y++)i=A.i[l[D]][y],T[i]||(l[++L]=i,T[i]=true,r[i]=n.ll(A,t,i,u,Q));++D;}for(P=0;P<A.r;P++)T[A.D[0][P]]&&n.ml(A,t,P,u,Q,r==null?A.D[0][P]:r[A.D[0][P]],r==null?A.D[1][P]:r[A.D[1][P]],false);n.cn(t,1),n.cn(t,2),t.T=0;},n.yo=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T;for(t.j=B(o.Q7,H,6,t.q,15,1),t.e=B(o.Q7,H,6,t.q,15,1),t.i=B(o.Q7,E1,7,t.q,0,2),t.k=B(o.Q7,E1,7,t.q,0,2),t.n=B(o.Q7,E1,7,t.q,0,2),t.o=B(o.Q7,H,6,t.f,15,1),l=B(o.Q7,H,6,t.q,15,1),D=0;D<t.r;D++)++l[t.D[0][D]],++l[t.D[1][D]];for(r=0;r<t.q;r++)t.i[r]=B(o.Q7,H,6,l[r],15,1),t.k[r]=B(o.Q7,H,6,l[r],15,1),t.n[r]=B(o.Q7,H,6,l[r],15,1);for(y=false,u=0;u<t.g;u++){if(T=n.fm(t,u),T==0){y=true;continue;}for(L=0;L<2;L++)e=t.D[L][u],A=t.e[e],t.n[e][A]=T,t.i[e][A]=t.D[1-L][u],t.k[e][A]=u,++t.e[e],++t.j[e],e<t.f&&(T>1?t.o[e]+=T-1:t.H[u]==8&&(t.o[e]=1));}for(Q=t.g;Q<t.r;Q++){if(T=n.fm(t,Q),T==0){y=true;continue;}for(L=0;L<2;L++)e=t.D[L][Q],A=t.e[e],t.n[e][A]=T,t.i[e][A]=t.D[1-L][Q],t.k[e][A]=Q,++t.e[e],t.D[1-L][Q]<t.f&&++t.j[e];}if(y){for(A=B(o.Q7,H,6,t.q,15,1),P=0;P<t.q;P++)A[P]=t.e[P];for(i=0;i<t.r;i++)if(T=n.fm(t,i),T==0)for(L=0;L<2;L++)e=t.D[L][i],t.n[e][A[e]]=T,t.i[e][A[e]]=t.D[1-L][i],t.k[e][A[e]]=i,++A[e];}},n.zo=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V;for(n.bw(t,1),e&&(A=true),Q=0;Q<t.r;Q++)if(l=n.fm(t,Q),l==1||l==2){if(t.s[t.D[0][Q]]>0&&t.s[t.D[1][Q]]<0)i=t.D[0][Q],D=t.D[1][Q];else if(t.s[t.D[0][Q]]<0&&t.s[t.D[1][Q]]>0)i=t.D[1][Q],D=t.D[0][Q];else continue;if(n.Lm(t,i)||n.Lm(t,D)||t.C[i]<9&&n.vp(t,i)>3||t.C[D]<9&&n.vp(t,D)>3)continue;y=n.op(t,i)!=0,t.s[i]-=1,t.s[D]+=1,y||(S=t.H[Q],l==1?t.H[Q]=2:t.H[Q]=4,(S==129||S==257)&&(V=t.D[0][Q],v=n.pq(t,V,false),t.D[0][v]!=V&&(t.D[1][v]=t.D[0][v],t.D[0][v]=V))),t.T=0;}for(p=0,g=0,c=0,P=0;P<t.q;P++)p+=t.s[P],t.s[P]<0&&!n.Pp(t,P)&&(++g,n.Hm(t,P)&&(c-=t.s[P]));if(!A&&p!=0)throw st(new h.cC("molecule's overall charges are not balanced"));for(n.bw(t,1),E=0,q=e?p+c:c,u=0;u<t.q;u++)t.s[u]>0&&!n.Op(t,u)&&n.Hm(t,u)&&(L=d.Math.min(n.op(t,u),t.s[u]),L!=0&&q>=L&&(p-=L,E-=L,q-=L,t.s[u]-=L,t.T&=1));if(k=e?p:E,k<0){for(b=B(o.Q7,H,6,g,15,1),g=0,i=0;i<t.q;i++)t.s[i]<0&&!n.Pp(t,i)&&(b[g++]=(t.C[i]<<22)+i);for(w.lAb(b,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),T=b.length-1;k<0&&T>=b.length-g;T--)r=b[T]&W1,n.Hm(t,r)&&(L=d.Math.min(-k,-t.s[r]),p+=L,k+=L,t.s[r]+=L,t.T&=1);}return p;},n.Ao=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c;for(e=0;e<t.q;e++)c=n.pm(t,e),c+n.nm(t,e,c,t.C[e])-n.vp(t,e)<=0&&!(t.s[e]==0&&(t.C[e]==5||(P=t.C[e],P==7||P==15||P==33)||(r=t.C[e],r==8||r==16||r==34||r==52||r==84)))&&(t.B[e]=J(t.B[e],-6145));if(t.S)return  false;for(D=false,A=0;A<t.f;A++)if(u=t.e[A]-t.j[A],u>0)for(S1(J(t.B[A],B1),0)&&(L=S1(J(t.B[A],X0),896)?3:S1(J(t.B[A],X0),384)?2:S1(J(t.B[A],128),128)?1:0,Q=(c=n.pm(t,A),c+n.nm(t,A,c,t.C[A])-n.vp(t,A)),t.s[A]==0&&S1(J(t.B[A],mA),0)&&t.C[A]!=6&&++Q,y=u,y>3-L&&(y=3-L),y>Q+u-L&&(y=Q+u-L),y>0&&(T=L==0?0:h1(J(t.B[A],X0),y),T=Pt(T,o.Kfb((y==3?7:u==2?3:1)<<7)),t.B[A]=J(t.B[A],-1921),t.B[A]=Pt(t.B[A],J(X0,T)))),l=t.j[A];l<t.e[A];l++)i=t.k[A][l],t.H[i]==1&&(t.C[t.i[A][l]]=-1,t.H[i]=512,D=true);return D&&n.kl(t),D;},n.Bo=function(t,A){var e,r,P,i;if(t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2){for(P=0;P<2;P++)if(e=n.Go(t,A,t.i[A][P]),e!=-1)for(i=0;i<t.j[e];i++)r=t.k[e][i],(t.H[r]==257||t.H[r]==129)&&t.D[0][r]==e&&(t.H[t.k[e][i]]=1);return;}if(t.o[A]==0||t.C[A]>=15)for(P=0;P<t.e[A];P++)r=t.k[A][P],(t.H[r]==257||t.H[r]==129)&&t.D[0][r]==A&&t.D[0][r]==A&&(t.H[r]=1);},n.Co=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T;for(n.bw(t,r?7:1),A.v=null,t.K&&n.On(A,true),Q=e.length,P==null&&(P=B(o.Q7,H,6,Q,15,1)),A.q=0,i=0;i<Q;i++)if(e[i]){if(P[i]=n.ll(t,A,i,0,0),t.C[i]==1&&(T=n.vp(t,i),T>1)){for(y=0;y<t.e[i];y++)if(!e[t.i[i][y]]){n.gn(A,P[i],T);break;}}}else P[i]=-1;for(L=B(o.Q7,H,6,t.r,15,1),U.Jrb(L,L.length,-1),A.r=0,l=0;l<t.r;l++)D=t.D[0][l],u=t.D[1][l],D<Q&&u<Q&&(e[D]&&e[u]?L[l]=n.ml(t,A,l,0,0,P==null?t.D[0][l]:P[t.D[0][l]],P==null?t.D[1][l]:P[t.D[1][l]],r):t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(e[D]&&(A.s[P[D]]+=t.s[D]<0?1:-1),e[u]&&(A.s[P[u]]+=t.s[u]<0?1:-1)));n.pl(t,A),t.d&&(A.T=0),A.T=0,n.cn(A,1),n.cn(A,2),A.q!=Q&&n.On(A,true),n.sq(t,A,Q,P,L),n.rq(t,A,Q,P),r&&n.Ge(new n.Ve(A),null,false);},n.Do=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T;for(n.bw(t,r?7:1),A.v=null,t.K&&n.On(A,true),P==null&&(P=B(o.Q7,H,6,t.q,15,1)),A.q=0,i=0;i<t.q;i++)for(P[i]=-1,L=0;L<t.j[i];L++)if(e[t.k[i][L]]){if(P[i]=n.ll(t,A,i,0,0),t.C[i]==1&&(T=n.vp(t,i),T>1)){for(y=0;y<t.e[i];y++)if(!e[t.k[i][y]]){n.gn(A,P[i],T);break;}}break;}for(l=B(o.Q7,H,6,t.r,15,1),A.r=0,Q=0;Q<t.r;Q++)e[Q]?l[Q]=n.ml(t,A,Q,0,0,P==null?t.D[0][Q]:P[t.D[0][Q]],P==null?t.D[1][Q]:P[t.D[1][Q]],r):(l[Q]=-1,D=t.D[0][Q],u=t.D[1][Q],P[D]==-1^P[u]==-1&&t.s[D]!=0&&t.s[u]!=0&&t.s[D]<0^t.s[u]<0&&(P[D]!=-1&&(A.s[P[D]]+=t.s[D]<0?1:-1),P[u]!=-1&&(A.s[P[u]]+=t.s[u]<0?1:-1)));return n.pl(t,A),t.d&&(A.T=0),A.T=0,n.cn(A,1),n.cn(A,2),A.q!=t.q&&n.On(A,true),n.sq(t,A,t.q,P,l),n.rq(t,A,t.q,P),r&&n.Ge(new n.Ve(A),null,false),P;},n.Eo=function(t,A){var e,r,P,i,D,u,Q,l,L,y;if((A&~t.T)!=0&&((t.T&1)==0&&(n.Np(t),n.yo(t),t.T|=1,t.K&&!n.vm(t)&&n.Ao(t)&&(n.Np(t),n.yo(t),t.T|=1)),(A&~t.T)!=0)){if((t.T&-7)!=0){for(r=0;r<t.f;r++)t.u[r]&=-15369;for(i=0;i<t.g;i++)t.F[i]&=-705;if((A&4)==0){n.Jo(t,1),t.T|=2;return;}for(n.Jo(t,7),P=0;P<t.f;P++)for(L=0;L<t.j[P];L++)if(Q=t.k[P][L],!(Q<t.g&&n.pt(t.p,Q)))for(u=t.i[P][L],y=0;y<t.j[u];y++)t.k[u][y]!=Q&&t.n[u][y]>1&&(t.C[t.i[u][y]]==6?t.u[P]|=Z1:!n.Up(t,t.k[u][y])&&n.Hm(t,t.i[u][y])&&(t.u[P]|=L0));for(;;){for(l=false,e=0;e<t.f;e++)if(t.o[e]>0&&(t.u[e]&L0)!=0&&!n.ot(t.p,e)){for(L=0;L<t.j[e];L++)if(t.n[e][L]>1)for(u=t.i[e][L],Q=t.k[e][L],y=0;y<t.j[u];y++)t.k[u][y]!=Q&&(D=t.i[u][y],(t.u[D]&L0)==0&&(t.u[D]|=L0,l=true));}if(!l)break;}}t.T|=6;}},n.Fo=function(t,A){var e,r,P,i,D;if(e=-1,t.o[A]==1){for(i=0;i<t.j[A];i++)if(t.n[A][i]==2){if(r=t.i[A][i],t.j[r]==2&&t.o[r]==2){for(D=0;D<2;D++)if(P=t.i[r][D],P!=A&&t.o[P]==1){e=r;break;}}break;}}return e;},n.Go=function(t,A,e){var r,P;for(r=A;t.j[e]==2&&t.o[e]==2&&e!=r;)P=e,e=t.i[e][0]==A?t.i[e][1]:t.i[e][0],A=P;return e==r?-1:e;},n.Ho=function(t,A){var e;if(t.j[A]==3&&A<t.f&&n.ot(t.p,A)&&(t.p&&A<t.f?n.dt(t.p,A):0)>=5){for(e=0;e<t.j[A];e++)if(n.Vp(t,t.k[A][e]))return t.k[A][e];}return  -1;},n.Io=function(t,A,e,r,P){var i,D,u,Q,l,L;if(n.bw(t,7),!((t.u[A]&V0)==0||e&&!(A<t.f&&n.ot(t.p,A))))for(Q=B(o.Q7,H,6,t.f,15,1),Q[0]=A,r[A]=true,u=0,l=0;u<=l;){for(L=0;L<t.j[Q[u]];L++)D=t.k[Q[u]][L],!P[D]&&(t.F[D]&64)!=0&&(!e||D<t.g&&n.pt(t.p,D))&&(P[D]=true,i=t.i[Q[u]][L],r[i]||(r[i]=true,Q[++l]=i));++u;}},n.Jo=function(t,A){var e,r,P,i,D,u,Q,l;for(t.p=new n.Dt(t,A),r=B(o.Q7,H,6,t.f,15,1),P=0;P<t.g;P++)n.ft(t.p,P)!=0&&(t.F[P]|=64,++r[t.D[0][P]],++r[t.D[1][P]]);for(e=0;e<t.f;e++)r[e]==2?t.u[e]|=f0:r[e]==3?t.u[e]|=B1:r[e]>3&&(t.u[e]|=V0);for(l=0;l<t.p.j.a.length;l++)for(D=n.ht(t.p,l),Q=n.it(t.p,l),u=D.length,i=0;i<u;i++)t.u[D[i]]|=8,t.F[Q[i]]|=128,t.H[Q[i]]==386&&(t.H[Q[i]]=2);},n.Ko=function(t){var A,e,r,P,i,D,u,Q;for(u=B(o.vfb,Tt,6,t.q,16,1),e=0;e<t.q;e++)u[e]=t.C[e]==1&&t.A[e]==0&&t.s[e]==0&&(t.u[e]&bA)==0&&(t.t==null||t.t[e]==null);for(Q=B(o.vfb,Tt,6,t.q,16,1),D=0;D<t.r;D++){if(r=t.D[0][D],P=t.D[1][D],n.fm(t,D)!=1){u[r]=false,u[P]=false;continue;}Q[r]&&(u[r]=false),Q[P]&&(u[P]=false),u[r]&&n.Lm(t,P)&&t.C[P]!=13&&(u[r]=false),u[P]&&n.Lm(t,r)&&t.C[r]!=13&&(u[P]=false),Q[r]=true,Q[P]=true;}for(i=0;i<t.r;i++)u[t.D[0][i]]&&u[t.D[1][i]]&&(u[t.D[0][i]]=false,u[t.D[1][i]]=false);for(A=0;A<t.q;A++)Q[A]||(u[A]=false);return u;},n.Lo=function(t,A){return t.e[A];},n.Mo=function(t,A){return t.i[A].length;},n.No=function(t,A){return t.e[A]-n.tp(t,A)+n.op(t,A);},n.Oo=function(t){var A,e;for(n.bw(t,7),A=0,e=0;e<t.p.j.a.length;e++)n.nt(t.p,e)&&++A;return A;},n.Po=function(t,A){var e,r;for(e=0,r=0;r<t.j[A];r++)n.Hm(t,t.i[A][r])&&!n.Cm(t,t.k[A][r])&&++e;return e;},n.Qo=function(t,A){return t.o[A];},n.Ro=function(t,A){return n.bw(t,7),t.o[A]==2&&t.j[A]==2?n.nq(t,A,false):n.pq(t,A,false);},n.So=function(t,A){var e;return e=t.u[A]&V0,e==0?0:e==f0?2:e==B1?3:4;},n.To=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;for(n.bw(t,7),i=B(o.vfb,Tt,6,t.g,16,1),y=B(o.vfb,Tt,6,t.g,16,1),b=B(o.Q7,H,6,t.f,15,1),D=0,u=1;u<t.j[A];u++)if(r=t.k[A][u],(t.F[r]&64)!=0){for(l=0;l<u;l++)if(P=t.k[A][l],(t.F[P]&64)!=0&&(y[r]=true,y[P]=true,c=n.yp(t,b,t.i[A][u],t.i[A][l],e-2,null,y),y[r]=false,y[P]=false,c!=-1)){for(Q=false,T=B(o.Q7,H,6,c,15,1),n.zp(t,b,T,c),L=0;L<c;L++)i[T[L]]||(i[T[L]]=true,Q=true);Q&&++D;}}return D;},n.Uo=function(t,A){return t.p&&A<t.f?n.dt(t.p,A):0;},n.Vo=function(t,A){return A?(n.bw(t,1),n.Wl(t,t.f,t.g,n.Qk)):n.Wl(t,t.q,t.r,n.Qk);},n.Wo=function(t){var A,e,r,P,i,D,u,Q;for(n.bw(t,1),u=B(o.P7,w0,6,t.f,15,1),r=B(o.Q7,H,6,t.f,15,1),Q=0;Q<t.f;Q++){for(r[0]=Q,P=B(o.Q7,H,6,t.f,15,1),P[Q]=1,e=0,i=0;e<=i;){for(D=0;D<t.j[r[e]];D++)A=t.i[r[e]][D],P[A]==0&&(P[A]=P[r[e]]+1,r[++i]=A,u[Q]+=P[A]-1);++e;}u[Q]/=i;}return u;},n.Xo=function(t,A,e){var r;for(r=0;r<t.i[A].length;r++)if(t.i[A][r]==e)return t.k[A][r];return  -1;},n.Yo=function(t,A){return t.p&&A<t.g?n.ft(t.p,A):0;},n.Zo=function(t,A,e){return t.i[A][e];},n.$o=function(t,A){return t.j[A];},n._o=function(t,A,e){return t.k[A][e];},n.ap=function(t,A,e){return t.n[A][e];},n.bp=function(t,A){var e,r;if(e=0,t.K)for(r=0;r<t.j[A];r++)n.Jm(t,t.i[A][r])&&++e;return e;},n.cp=function(t,A){return t.e[A]-t.j[A];},n.dp=function(t,A,e,r,P){var i,D,u,Q;if(D=t.e[A],t.o[A]!=0||A<t.f&&n.ot(t.p,A)||t.j[A]<3||D>4)return  false;for(Q=B(o.vfb,Tt,6,4,16,1),u=0;u<D;u++){if(i=3.9269908169872414-r[u],d.Math.abs(lP-i%v0)>.0872664675116539||(P[u]=3&Mt(i/v0),Q[P[u]]))return  false;if(Q[P[u]]=true,(P[u]&1)==0){if(t.H[t.k[A][e[u]]]!=1)return  false;}else if(!n.Sm(t,t.k[A][e[u]],A))return  false;}return Q[0]&&Q[2];},n.ep=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T;if((t.p&&A<t.f?n.dt(t.p,A):0)>24||(i=t.e[A],P==null&&(P=B(o.Q7,H,6,i,15,1)),!n.dp(t,A,e,r,P)))return 3;for(Q=-1,l=0;l<i;l++)if((P[l]&1)==1){if(D=t.H[t.k[A][e[l]]],Q!=-1&&Q!=D)return 3;Q=D;}return L=d.Math.abs(P[0]-P[1])==2?1:0,u=P[L]-P[L+1],T=d.Math.abs(u)==3^P[L]<P[L+1],y=i==3||(P[3]&1)==1,T^y^Q==129?1:2;},n.fp=function(t,A,e){var r,P,i,D;for(D=D=B(o.vfb,Tt,6,t.q,16,1),i=n.gp(t,A,e,D),P=B(o.Q7,H,6,i,15,1),i=0,r=0;r<t.q;r++)D[r]&&(P[i++]=r);return P;},n.gp=function(t,A,e,r){var P,i,D,u,Q,l,L;for(n.bw(t,1),r==null&&(r=B(o.vfb,Tt,6,t.q,16,1)),Q=B(o.Q7,H,6,t.q,15,1),Q[0]=A,r[A]=true,D=0,l=0,u=1;D<=l;){for(i=e?n.Mo(t,Q[D]):t.e[Q[D]],L=0;L<i;L++)P=t.i[Q[D]][L],P<r.length&&!r[P]&&(Q[++l]=P,r[P]=true,++u);++D;}return u;},n.hp=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T;for(n.bw(t,1),i=0;i<t.q;i++)A[i]=-1;for(l=0,P=0;P<t.q;P++)if(A[P]==-1&&(!e||(t.u[P]&r0)!=0)){for(A[P]=l,L=B(o.Q7,H,6,t.q,15,1),L[0]=P,Q=0,y=0;Q<=y;){for(u=r?n.Mo(t,L[Q]):t.e[L[Q]],T=0;T<u;T++)D=t.i[L[Q]][T],A[D]==-1&&(!e||(t.u[D]&r0)!=0)&&(L[++y]=D,A[D]=l);++Q;}++l;}return l;},n.ip=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T;for(n.bw(t,1),i=0;i<t.q;i++)A[i]=-1;for(l=0,P=0;P<t.q;P++)if(A[P]==-1){for(A[P]=l,L=B(o.Q7,H,6,t.q,15,1),L[0]=P,Q=0,y=0;Q<=y;){for(u=r?n.Mo(t,L[Q]):t.e[L[Q]],T=0;T<u;T++)D=t.i[L[Q]][T],A[D]==-1&&!e[t.k[L[Q]][T]]&&(L[++y]=D,A[D]=l);++Q;}++l;}return l;},n.jp=function(t,A){var e;return e=n.pm(t,A),e+n.nm(t,A,e,t.C[A])-n.vp(t,A);},n.kp=function(t,A){var e,r,P,i,D,u;for(i=B(o.Q7,H,6,t.q,15,1),r=0;r<t.q;r++)i[r]=r;P=t.q;do--P;while(P>=0&&A[P]);for(e=0;e<P;e++)if(A[i[e]]){u=i[e],i[e]=i[P],i[P]=u,D=A[e],A[e]=A[P],A[P]=D;do--P;while(A[P]);}return i;},n.lp=function(t){return n.kp(t,n.Ko(t));},n.mp=function(t,A,e){var r,P,i;if(P=n.vp(t,A),P-=n.nm(t,A,P,t.C[A]),e&&(P-=t.e[A]-t.j[A]),i=n.mo(t.C[A]),P<=i[0])return  -1;for(r=1;r<i.length;r++)if(i[r]>=P)return i[r];return P;},n.np=function(t){var A,e;if(t.K)return 0;for(n.bw(t,1),e=0,A=0;A<t.f;A++)e+=n.op(t,A);return e;},n.op=function(t,A){var e,r,P,i,D,u,Q;if(t.K&&S1(J(t.B[A],B1),0)||!n.Aq(t,A))return 0;if(t.C[A]==0||h.akb("*",t.t==null||t.t[A]==null?null:h.$jb((i=t.t[A],q1.Cnb(),i))))return t.e[A]==0?1:0;for(n.bw(t,1),u=0,P=0;P<t.e[A];P++)u+=t.n[A][P];if(t.K){for(e=1,r=0;r<t.j[A];r++)t.H[t.k[A][r]]==8&&++e;u+=e>>1;}if(u-=n.nm(t,A,u,t.C[A]),D=((t.u[A]&bA)>>>27)-1,D==-1)for(Q=n.mo(t.C[A]),D=Q[0],r=1;D<u&&r<Q.length;r++)D=Q[r];return d.Math.max(0,D-u);},n.pp=function(t,A){var e,r,P,i,D;if(!t.K||t.v==null||t.v[A]==null)return n.qp(t,A,t.C[A]);for(D=0,r=t.v[A],P=0,i=r.length;P<i;++P)e=r[P],D=d.Math.max(D,n.qp(t,A,e));return D;},n.qp=function(t,A,e){var r,P,i,D,u;if(i=n.vp(t,A),r=n.nm(t,A,i,e),D=((t.u[A]&bA)>>>27)-1,D==-1){for(u=n.mo(t.C[A]),P=0;P<u.length-1&&i>u[P]+r;)++P;D=u[P];}return D+r-i;},n.rp=function(t,A){return t.i[A].length-t.e[A];},n.sp=function(t){var A,e,r,P;for(n.bw(t,1),P=0,A=0;A<t.q;A++)r=t.A[A]!=0?t.A[A]:n.Pk[t.C[A]],P+=r+n.op(t,A)*n.Pk[1],t.C[A]>=171&&t.C[A]<=190&&(e=t.e[A],e>2&&(P-=(e-2)*n.Pk[1]));return P;},n.tp=function(t,A){var e,r;for(e=t.j[A],r=0;r<t.j[A];r++)t.C[t.i[A][r]]==1&&--e;return e;},n.up=function(t,A){return t.j[A]-n.bp(t,A);},n.vp=function(t,A){var e,r,P,i,D,u;for(t.$b(1),D=false,r=false,u=0,P=0;P<t.e[A];P++)(!t.K||!n.Jm(t,t.i[A][P]))&&(i=t.n[A][P],u+=i,i>1&&(D=true),e=t.k[A][P],t.H[e]==8&&(r=true));return r&&!D&&++u,u;},n.wp=function(t,A,e){var r,P,i;for(P=0,i=0;i<t.j[A];i++)r=t.i[A][i],r!=e&&t.j[r]>2&&++P;return P;},n.xp=function(t,A){var e,r,P,i,D,u,Q,l,L,y;if(n.to(t.C[A]))return 0;if(t.C[A]==1)return t.j[A]==0?0:n.ro(t.C[t.i[A][0]])?-1:1;if(y=t.s[A],Q=t.e[A]-n.tp(t,A)+n.op(t,A),t.C[A]!=1&&t.C[A]!=15&&(n.ro(t.C[A])?y+=Q:y-=Q),t.j[A]==0)return y;for(L=0;L<t.j[A];L++)i=t.i[A][L],e=n.Yp(t,t.k[A][L])?1:t.n[A][L],n.ro(t.C[A])?n.ro(t.C[i])||(y+=e):n.ro(t.C[i])?y-=e:t.C[A]!=t.C[i]&&(n.Xs(t.C[A]).a<n.Xs(t.C[i]).a?y+=e:y-=e);if(A<t.f&&n.tt(t.p,A)){for(r=0,P=0,u=0,l=0;l<t.j[A];l++)n.Yp(t,t.k[A][l])&&(D=t.C[t.i[A][l]],D==6?++P:n.qo(D)?++u:++r);t.C[A]==6?r>1?--y:u>1&&++y:n.Hm(t,A)?r+P>1&&--y:u+P>1&&++y;}return y;},n.yp=function(t,A,e,r,P,i,D){var u,Q,l,L,y,T,c,b,g;if(e==r)return A[0]=e,0;for(t.$b(1),L=B(o.Q7,H,6,t.q,15,1),l=B(o.Q7,H,6,t.q,15,1),g=B(o.Q7,H,6,t.q,15,1),l[0]=e,L[e]=1,Q=0,y=0;Q<=y&&L[l[Q]]<=P;){for(b=l[Q],T=0;T<t.e[b];T++)if(D==null||D.length<=t.k[b][T]||!D[t.k[b][T]]){if(u=t.i[b][T],u==r){for(c=L[b],A[c]=u,A[--c]=b;c>0;)A[c-1]=g[A[c]],--c;return L[b];}L[u]==0&&(i==null||i.length<=u||!i[u])&&(l[++y]=u,L[u]=L[b]+1,g[u]=b);}++Q;}return  -1;},n.zp=function(t,A,e,r){var P,i;for(n.bw(t,1),P=0;P<r;P++)for(i=0;i<t.e[A[P]];i++)if(t.i[A[P]][i]==A[P+1]){e[P]=t.k[A[P]][i];break;}},n.Ap=function(t,A,e){var r,P,i,D,u,Q;if(A==e)return 0;for(n.bw(t,1),D=B(o.Q7,H,6,t.q,15,1),i=B(o.Q7,H,6,t.q,15,1),i[0]=A,D[A]=1,P=0,u=0;P<=u;){for(Q=0;Q<t.e[i[P]];Q++){if(r=t.i[i[P]][Q],r==e)return D[i[P]];D[r]==0&&(i[++u]=r,D[r]=D[i[P]]+1);}++P;}return  -1;},n.Bp=function(t,A,e,r,P){var i,D,u,Q,l,L;if(A==e)return 0;for(t.$b(1),Q=B(o.Q7,H,6,t.q,15,1),u=B(o.Q7,H,6,t.q,15,1),u[0]=A,Q[A]=1,D=0,l=0;D<=l&&Q[u[D]]<=r;){for(L=0;L<t.e[u[D]];L++){if(i=t.i[u[D]][L],i==e)return Q[u[D]];Q[i]==0&&(P==null||P.length<=i||!P[i])&&(u[++l]=i,Q[i]=Q[u[D]]+1);}++D;}return  -1;},n.Cp=function(t,A){return t.e[A]-t.j[A]+n.op(t,A);},n.Dp=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b;for(b=B(o.Q7,H,6,16,15,1),e=B(o.O7,Ft,6,16,15,1),i=B(o.O7,Ft,6,2,15,1),r=0,L=0;L<2;L++)for(P=t.D[L][A],y=0;y<t.j[P];y++)if(Q=t.k[P][y],Q!=A){if(r==4)return 0;u=t.i[P][y],b[r]=16,t.o[u]!=0&&(b[r]+=(t.u[u]&V0)!=0?1:4),(t.F[A]&64)!=0&&(t.F[Q]&64)!=0&&(T=n.lt(t.p,A,Q),T!=-1&&(b[r]+=n.nt(t.p,T)?64:6)),e[r++]=n.no(t.J[P].a,t.J[P].b,t.J[u].a,t.J[u].b);}for(i[0]=n.Yl(t,t.D[0][A],t.D[1][A]),i[0]<0?(i[1]=i[0]+A1,D=false):(i[1]=i[0],i[0]=i[1]-A1,D=true),c=0,l=0;l<r;l++)e[l]>i[0]&&e[l]<i[1]?c-=b[l]:c+=b[l];return D?-c:c;},n.Ep=function(t){return n.bw(t,7),t.p;},n.Fp=function(t){var A,e,r,P,i,D,u,Q,l;for(l=0,n.bw(t,7),r=0;r<t.g;r++)if(n.fm(t,r)==1&&(t.F[r]&64)==0){for(u=true,D=0;D<2;D++){if(A=t.D[D][r],t.j[A]==1){u=false;break;}if(t.C[A]==7&&!(A<t.f&&n.ot(t.p,A))){for(e=t.D[1-D][r],Q=0;Q<t.j[e];Q++)if(P=t.i[e][Q],i=t.k[e][Q],i!=r&&n.fm(t,i)>1&&!(P<t.f&&n.ot(t.p,P))&&n.Hm(t,P)){u=false;break;}}}u&&!n.cq(t,r)&&++l;}return l;},n.Gp=function(t,A,e){var r,P,i,D,u,Q;for(P=0,i=0;i<t.j[A];i++)if(r=t.i[A][i],r!=e){for(D=0,u=0;u<t.j[r];u++)Q=t.i[r][u],Q!=A&&n.Up(t,t.k[r][u])&&t.j[Q]>2&&++D;D==2&&++P;}return P;},n.Hp=function(t,A){var e,r,P,i;for(e=t.e[A],i=B(o.Q7,H,6,e,15,1),P=0;P<e;P++)i[P]=(t.i[A][P]<<16)+P;for(w.lAb(i,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),r=0;r<e;r++)i[r]&=e1;return i;},n.Ip=function(t,A){var e,r;if(n.bw(t,1),t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2){for(e=0;e<2;e++)for(r=0;r<t.e[t.i[A][e]];r++)if(n.Sm(t,t.k[t.i[A][e]][r],t.i[A][e]))return t.k[t.i[A][e]][r];}else for(e=0;e<t.e[A];e++)if(n.Sm(t,t.k[A][e],A))return t.k[A][e];return  -1;},n.Jp=function(t,A,e){return n.fm(t,A)!=1?0:16-t.e[e]+16*d.Math.max(0,d.Math.min(9,t.p&&A<t.g?n.ft(t.p,A):0)-2)+((t.H[A]&384)==0||t.D[0][A]!=e?LA:0)+((t.u[e]&3)==0?Z1:0)+(t.C[e]==1?B1:0)+(t.e[e]==1?f0:0)+((t.F[A]&64)!=0?0:512)+(t.C[e]!=6?256:0);},n.Kp=function(t,A,e,r,P,i){var D,u,Q,l,L;for(n.bw(t,1),P&&(n.jl(P),P.d=null,P.K=false),Q=B(o.Q7,H,6,t.q,15,1),r==null?r=B(o.vfb,Tt,6,t.q,16,1):U.Nrb(r,r.length),Q[0]=A,Q[1]=e,r[A]=true,r[e]=true,u=1,l=1;u<=l;){for(L=0;L<t.e[Q[u]];L++){if(D=t.i[Q[u]][L],D==A&&u!=1)return  -1;r[D]||(r[D]=true,Q[++l]=D);}++u;}return P&&(i==null&&(i=B(o.Q7,H,6,r.length,15,1)),n.Co(t,P,r,false,i),n.fl(P,i[A],0,0,-1,0)),r[A]=false,l;},n.Lp=function(t,A,e){var r,P,i,D,u,Q;for(n.bw(t,1),i=B(o.Q7,H,6,t.f,15,1),Q=B(o.vfb,Tt,6,t.f,16,1),i[0]=A,i[1]=e,Q[A]=true,Q[e]=true,P=1,D=1;P<=D;){for(u=0;u<t.j[i[P]];u++){if(r=t.i[i[P]][u],r==A&&P!=1)return  -1;Q[r]||(Q[r]=true,i[++D]=r);}++P;}return D;},n.Mp=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c;if(n.fm(t,e)!=2&&!(e<t.g&&n.pt(t.p,e))||(c=t.F[e]&3,c!=1&&c!=2))return  -1;for(Q=0;Q<2;Q++){for(r=t.D[Q][e],P=t.D[1-Q][e],T=-1,D=false,L=0;L<t.j[r];L++)i=t.i[r][L],i!=P&&(i==A?D=true:T=i);if(D){for(y=-1,u=-1,l=0;l<t.j[P];l++)i=t.i[P][l],i!=r&&(y==-1?y=i:i>y?u=i:(u=y,y=i));return t.j[r]==2?t.j[P]==2?c==2?y:-1:c==2?y:u:t.j[P]==2?c==2^A<T?-1:y:c==2^A<T?u:y;}}return  -1;},n.Np=function(t){var A,e,r,P,i,D,u,Q,l,L;D=n.Ko(t),Q=t.q;do--Q;while(Q>=0&&D[Q]);for(A=0;A<Q;A++)if(D[A]){n.Zn(t,A,Q),L=D[A],D[A]=D[Q],D[Q]=L;do--Q;while(D[Q]);}if(t.f=Q+1,t.q==t.f){t.g=t.r;return;}for(u=B(o.vfb,Tt,6,t.r,16,1),i=0;i<t.r;i++)e=t.D[0][i],r=t.D[1][i],(D[e]||D[r])&&(u[i]=true);l=t.r;do--l;while(l>=0&&u[l]);for(P=0;P<l;P++)if(u[P]){n.$n(t,P,l),u[P]=false;do--l;while(u[l]);}t.g=l+1;},n.Op=function(t,A){var e;for(e=0;e<t.j[A];e++)if(t.s[t.i[A][e]]<0)return  true;return  false;},n.Pp=function(t,A){var e;for(e=0;e<t.j[A];e++)if(t.s[t.i[A][e]]>0)return  true;return  false;},n.Qp=function(t,A){var e;return e=t.C[A],e==3||e==11||e==19||e==37||e==55||e==87;},n.Rp=function(t,A){return (t.u[A]&Z1)!=0;},n.Sp=function(t,A){var e,r,P,i,D,u;for(n.bw(t,1),D=0;D<2;D++)if(e=t.D[D][A],t.C[e]==7){for(r=t.D[1-D][A],u=0;u<t.j[r];u++)if(P=t.i[r][u],i=t.k[r][u],(t.C[P]==7||t.C[P]==8||t.C[P]==16)&&n.fm(t,i)>=2)return  true;}return  false;},n.Tp=function(t,A){return A<t.f&&n.ot(t.p,A);},n.Up=function(t,A){return A<t.g&&n.pt(t.p,A);},n.Vp=function(t,A){var e,r,P,i,D,u;return t.H[A]!=1||A<t.g&&n.pt(t.p,A)||(t.F[A]&64)!=0&&(t.p&&A<t.g?n.ft(t.p,A):0)<7||(e=t.D[0][A],!(e<t.f&&n.ot(t.p,e))||(t.p&&e<t.f?n.dt(t.p,e):0)<5)||(r=t.D[1][A],!(r<t.f&&n.ot(t.p,r))||(t.p&&r<t.f?n.dt(t.p,r):0)<5)?false:(P=n.wp(t,e,r),i=n.wp(t,r,e),(t.p&&e<t.f?n.dt(t.p,e):0)>5&&(t.p&&r<t.f?n.dt(t.p,r):0)>5?P+i>2:(D=n.Gp(t,e,r),u=n.Gp(t,r,e),P==2&&u>=1||i==2&&D>=1||D==2&&(i>=1||u>=1)||u==2&&(P>=1||D>=1)));},n.Wp=function(t,A){return t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2&&t.C[A]<=7;},n.Xp=function(t,A){var e;return e=t.C[A],e==8||e==16||e==34||e==52||e==84;},n.Yp=function(t,A){return A<t.g&&(n.ut(t.p,A)||t.H[A]==8);},n.Zp=function(t,A){var e;return e=t.C[A],e==4||e==12||e==20||e==38||e==56||e==88;},n.$p=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g;if(t.C[A]!=7||t.j[A]==4)return  false;if(A<t.f&&n.ot(t.p,A)||t.o[A]!=0||pt(J(t.B[A],NA),0))return  true;if(t.s[A]==1)return  false;for(Q=0;Q<t.j[A];Q++)if(n.Yo(t,t.k[A][Q])==3)return  false;for(D=0,l=0;l<t.j[A];l++)t.n[A][l]==1&&(r=t.C[t.i[A][l]],(r==8||r==9||r==17)&&++D);if(D==0){for(u=0;u<t.j[A];u++)if(P=t.i[A][u],t.o[P]!=0){if(P<t.f&&n.ot(t.p,P)){if((t.p&&P<t.f?n.dt(t.p,P):0)>=5){for(g=0,T=0;T<t.j[P];T++)b=t.i[P][T],b!=A&&n.tp(t,b)>=3&&++g;if(c=n.tp(t,A),g==2&&c>=2||g==1&&c==3)continue;}return !e||!n.eq(t,A);}for(y=0;y<t.j[P];y++)if(t.n[P][y]==2||n.Up(t,t.k[P][y]))return !e||!n.eq(t,A);}}if(D<2)for(u=0;u<t.j[A];u++){for(P=t.i[A][u],L=false,i=false,y=0;y<t.j[P];y++)t.i[P][y]!=A&&(t.n[P][y]!=1&&(t.C[t.i[P][y]]==7||t.C[t.i[P][y]]==8||t.C[t.i[P][y]]==16)&&(L=true),t.n[P][y]==1&&t.C[t.i[P][y]]==7&&(i=true));if(L&&(!i||D==0))return !e||!n.eq(t,A);}return  false;},n._p=function(t,A){var e;return e=t.C[A],e==9||e==17||e==35||e==53||e==85;},n.aq=function(t,A){return A<t.f&&n.vt(t.p,A);},n.bq=function(t,A){var e;return e=t.C[A],e==7||e==15||e==33;},n.cq=function(t,A){var e,r,P,i,D,u;if(n.fm(t,A)!=1)return  false;for(i=0;i<2;i++){for(e=t.D[i][A],u=t.D[1-i][A];t.o[e]==2&&t.j[e]==2&&t.C[e]<10;)for(D=0;D<2;D++)if(r=t.i[e][D],r!=u){if(t.j[r]==1||(P=t.k[e][D],n.fm(t,P)==1&&P<A))return  true;u=e,e=r;break;}if(t.j[e]==1)return  true;}return  false;},n.dq=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b,g,k,v,S;if(T=B(o.vfb,Tt,6,t.g,16,1),T[P]=true,c=B(o.Q7,H,6,11,15,1),b=n.yp(t,c,r,A,10,null,T),b==-1)return  false;for(D=1;!n.qt(t.p,e,c[D]);)++D;if(i=b-D,u=c[D],S=n.kt(t.p,e),S==6&&i==2&&D==3&&n.So(t,c[1])>=3){for(v=n.ht(t.p,e),y=0;y<6;y++)if(A==v[y]){if(k=n.Ct(t.p,e,u==v[n.Ct(t.p,e,y+2)]?y-2:y+2),g=v[k],L=t.u[g]&V0,(L==0?0:L==f0?2:L==B1?3:4)>=3&&n.Bp(t,c[1],g,2,null)==2)return  true;break;}}if(Q=t.o[u]==1||u<t.f&&n.ot(t.p,u)||n.$p(t,u,false),l=!Q&&t.C[u]==7&&t.s[u]!=1,i==1)return !Q&&!l&&S<=4&&D<=3;switch(S){case 4:if(!Q&&!l&&D<=4)return  true;break;case 5:if(l){if(D<=3)return  true;}else if(!Q&&D<=4)return  true;break;case 6:if(i==2){if(Q){if(D<=4)return  true;}else if(!l&&D<=3)return  true;}else if(i==3){if(Q){if(D<=6)return  true;}else if(D<=4)return  true;}break;case 7:if(i==3&&D<=3)return  true;}return  false;},n.eq=function(t,A){var e,r,P,i,D;if(A<t.f&&n.ot(t.p,A)||t.o[A]!=0||pt(J(t.B[A],NA),0)||(r=t.u[A]&V0,(r==0?0:r==f0?2:r==B1?3:4)<3)||(D=t.p&&A<t.f?n.dt(t.p,A):0,D>7))return  false;for(i=0;i<t.p.j.a.length&&!(n.kt(t.p,i)==D&&n.qt(t.p,i,A));)++i;if(i>=f0&&i==t.p.j.a.length)return  false;for(P=0;P<t.j[A];P++)if(e=t.k[A][P],!n.rt(t.p,i,e)&&n.dq(t,A,i,t.i[A][P],e))return  true;return  false;},n.fq=function(t,A){return (t.u[A]&V0)!=0;},n.gq=function(t,A){return (t.F[A]&64)!=0;},n.hq=function(t,A){return t.C[A]==1&&t.A[A]==0&&t.s[A]==0&&(t.u[A]&bA)==0&&(t.t==null||t.t[A]==null);},n.iq=function(t,A){return (t.u[A]&8)!=0;},n.jq=function(t,A){return (t.F[A]&128)!=0;},n.kq=function(t,A){return (t.u[A]&L0)!=0;},n.lq=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(n.bw(t,1),n.mq(t),l=false,e=0;e<t.f;e++)if(t.C[e]==7&&t.s[e]==0){if(y=n.vp(t,e),y==4){for(L=0;L<t.j[e];L++)if(u=t.i[e][L],t.n[e][L]==1&&t.C[u]==8&&t.j[u]==1&&t.s[u]==0){l=true,++t.s[e],--t.s[u];break;}}else if(y==5)for(L=0;L<t.j[e];L++){if(u=t.i[e][L],Q=t.k[e][L],t.n[e][L]==2&&t.C[u]==8){l=true,++t.s[e],--t.s[u],t.H[Q]=1;break;}if(t.n[e][L]==3&&t.C[u]==7){l=true,++t.s[e],--t.s[u],t.H[Q]=2;break;}}}for(D=false,i=0;i<t.g;i++)for(L=0;L<2;L++)if(n.Hm(t,t.D[L][i])){A=t.D[1-L][i],P=t.C[A],(P==3||P==11||P==19||P==37||P==55||P==87||(r=t.C[A],r==4||r==12||r==20||r==38||r==56||r==88))&&(n.fm(t,i)==1?(++t.s[A],--t.s[t.D[L][i]],t.H[i]=512,D=true):t.H[i]==16&&(t.H[i]=512,D=true));break;}return D&&(n.kl(t),l=true),l&&(t.T=0),l;},n.mq=function(t){var A;for(A=0;A<t.g;A++)if(t.H[A]==8)return n.Ge(new n.Ve(t),null,false);return  false;},n.nq=function(t,A,e){var r,P,i,D,u,Q,l,L;for(l=-1,P=0,u=0;u<2;u++)for(r=t.i[A][u],Q=0;Q<t.e[r];Q++)i=t.i[r][Q],i!=A&&(D=t.k[r][Q],L=n.Jp(t,D,i),P<L&&(!e||!(t.H[D]==257||t.H[D]==129))&&(P=L,l=D));return l;},n.oq=function(t,A){var e,r,P,i,D,u,Q,l;for(Q=-1,r=0,D=0;D<2;D++)for(e=t.D[D][A],u=0;u<t.e[e];u++)P=t.i[e][u],P!=t.D[1-D][A]&&(i=t.k[e][u],l=n.Jp(t,i,P),r<l&&(r=l,Q=i));return Q;},n.pq=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;for(r=t.e[A],P=B(o.O7,Ft,6,r,15,1),c=0;c<r;c++)P[c]=n.Yl(t,A,t.i[A][c]);for(b=1;b<r;b++)for(g=0;g<b;g++)if(i=d.Math.abs(n.oo(P[b],P[g])),i<.08||i>J8){for(D=0,u=0,k=0;k<r;k++)k!=b&&k!=g&&(D+=d.Math.abs(Lt.m4(P[b],P[k])),u+=d.Math.abs(Lt.m4(P[g],P[k])));if(l=D<u?t.k[A][b]:t.k[A][g],n.fm(t,l)==1&&(!e||!(t.H[l]==257||t.H[l]==129)))return l;}for(v=-1,Q=0,T=0;T<r;T++)L=t.i[A][T],y=t.k[A][T],S=n.Jp(t,y,L),Q<S&&(!e||!(t.H[y]==257||t.H[y]==129))&&(Q=S,v=y);return v;},n.qq=function(t,A){var e,r,P,i,D,u;for(n.bw(t,A?15:1),t.q=t.f,t.r=t.g,u=B(o.vfb,Tt,6,t.f,16,1),P=0;P<t.f;P++)t.e[P]!=t.j[P]&&(e=n.mp(t,P,false),u[P]=t.e[P]==t.j[P]+1&&n.Rm(t,t.k[P][t.j[P]]),t.e[P]=t.j[P],e!=-1&&(D=n.mp(t,P,true),e!=D&&(i=((t.u[P]&bA)>>>27)-1,(i==-1||i<e)&&n.gn(t,P,e))));if(A)for(r=0;r<t.f;r++)u[r]&&n.wq(t,r);t.T=0;},n.rq=function(t,A,e,r){var P,i,D,u;for(n.bw(A,1),i=0;i<e;i++)r[i]!=-1&&t.C[i]!=1&&t.e[i]!=n.Lo(A,r[i])&&(P=n.mp(t,i,false),P!=-1&&(u=n.mp(A,r[i],false),P!=u&&(D=n.zl(A,r[i]),(D==-1||D<P)&&n.gn(A,r[i],P))));},n.sq=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b,g;for(u=0;u<e;u++)if(r[u]!=-1&&t.e[u]>3&&(t.u[u]&q0)!=0){for(g=0,T=-1,y=-1,L=0;L<t.e[u];L++)P[t.k[u][L]]!=-1?++g:t.n[u][L]==1&&n.Rm(t,t.k[u][L])&&t.D[0][t.k[u][L]]==u&&(T=t.k[u][L],y=t.i[u][L]);if(T!=-1&&g>=3){for(i=n.no(t.J[u].a,t.J[u].b,t.J[y].a,t.J[y].b),c=10,b=-1,l=0;l<t.e[u];l++)t.n[u][l]==1&&(!n.Rm(t,t.k[u][l])||t.D[0][t.k[u][l]]==u)&&P[t.k[u][l]]!=-1&&(D=d.Math.abs(n.oo(i,n.Yl(t,u,t.i[u][l]))),c>D&&(c=D,b=t.k[u][l]));b!=-1&&(Q=P[b],n.Mn(A,Q,t.H[b]==257?129:257),t.D[0][b]!=u&&(n.Dn(A,1,Q,r[t.D[0][b]]),n.Dn(A,0,Q,r[u])));}}},n.tq=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G;if(t.j[A]!=2||t.n[A][0]!=2||t.n[A][1]!=2||t.j[t.i[A][0]]<2||t.j[t.i[A][1]]<2||t.o[t.i[A][0]]!=1||t.o[t.i[A][1]]!=1){n.un(t,A,0,false);return;}for(q=-1,E=-1,p=-1,k=-1,i=0,y=0;y<2;y++)for(r=t.i[A][y],b=0;b<t.e[r];b++)D=t.i[r][b],D!=A&&(u=t.k[r][b],V=n.Jp(t,u,D),i<V&&(i=V,E=D,q=u,p=r,k=t.i[A][1-y]));if(E!=-1){for(T=0;T<2;T++)for(r=t.i[A][T],b=0;b<t.e[r];b++)D=t.i[r][b],u=t.k[r][b],D!=A&&t.D[0][u]==r&&(t.H[u]=1);for(t.D[1][q]!=E&&(t.D[0][q]=t.D[1][q],t.D[1][q]=E),Q=qt,c=0;c<t.j[p];c++)D=t.i[p][c],D!=A&&Q>D&&(Q=D);for(v=B(o.Q7,H,6,2,15,1),S=0,L=0;L<t.j[k];L++)D=t.i[k][L],D!=A&&(v[S++]=D);e=n.no(t.J[A].a,t.J[A].b,t.J[k].a,t.J[k].b),P=0,S==2?(v[0]>v[1]&&(G=v[0],v[0]=v[1],v[1]=G),l=n.oo(e,n.Yl(t,k,v[0])),g=n.oo(e,n.Yl(t,k,v[1])),P=l-g):P=n.oo(e,n.Yl(t,k,v[0])),P<0^(t.u[A]&3)==1^Q==E?t.H[q]=257:t.H[q]=129;}},n.uq=function(t,A,e,r){var P,i,D,u,Q,l;if(P=t.e[A],u=B(o.Q7,H,6,P,15,1),l=n.ep(t,A,e,r,u),l==3)return  false;for(D=(t.u[A]&3)==l?257:129,Q=0;Q<P;Q++)(u[Q]&1)==1&&(i=t.k[A][e[Q]],t.H[i]=D,t.D[0][i]!=A&&(t.D[1][i]=t.D[0][i],t.D[0][i]=A));return  true;},n.vq=function(t,A){t.T|=248&(8|A);},n.wq=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k;if(n.Bo(t,A),!((t.u[A]&3)==0||(t.u[A]&3)==3)){if(t.o[A]==2&&t.j[A]==2){n.tq(t,A);return;}if(t.j[A]<3||t.j[A]>4){n.un(t,A,0,false);return;}for(e=t.e[A],b=false,D=0;D<e;D++)if(n.fm(t,t.k[A][D])==1){b=true;break;}if(b){for(g=n.Hp(t,A),r=B(o.O7,Ft,6,e,15,1),u=0;u<e;u++)r[u]=n.Yl(t,t.i[A][g[u]],A);for(Q=0;Q<e;Q++)t.D[0][t.k[A][Q]]==A&&n.fm(t,t.k[A][Q])==1&&(t.H[t.k[A][Q]]=1);if(!((t.p&&A<t.f?n.dt(t.p,A):0)<=24&&n.uq(t,A,g,r))&&(T=n.pq(t,A,true),T!=-1)){for(t.D[0][T]!=A&&(t.D[1][T]=t.D[0][T],t.D[0][T]=A),c=-1,l=0;l<e;l++)if(T==t.k[A][g[l]]){c=l;break;}for(k=j(F(o.Q7,2),E1,7,0,[j(F(o.Q7,1),H,6,15,[2,1,2,1]),j(F(o.Q7,1),H,6,15,[1,2,2,1]),j(F(o.Q7,1),H,6,15,[1,1,2,2]),j(F(o.Q7,1),H,6,15,[2,1,1,2]),j(F(o.Q7,1),H,6,15,[2,2,1,1]),j(F(o.Q7,1),H,6,15,[1,2,1,2])]),i=1;i<e;i++)r[i]<r[0]&&(r[i]+=N1);if(e==3){switch(L=false,c){case 0:L=r[1]<r[2]&&r[2]-r[1]<A1||r[1]>r[2]&&r[1]-r[2]>A1;break;case 1:L=r[2]-r[0]>A1;break;case 2:L=r[1]-r[0]<A1;}P=(t.u[A]&3)==1^L?257:129;}else y=0,r[1]<=r[2]&&r[2]<=r[3]?y=0:r[1]<=r[3]&&r[3]<=r[2]?y=1:r[2]<=r[1]&&r[1]<=r[3]?y=2:r[2]<=r[3]&&r[3]<=r[1]?y=3:r[3]<=r[1]&&r[1]<=r[2]?y=4:r[3]<=r[2]&&r[2]<=r[1]&&(y=5),P=(t.u[A]&3)==1^k[y][c]==1?129:257;t.H[T]=P;}}}},n.xq=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G;if(!((t.F[A]&3)==0||(t.F[A]&3)==3||!n.Vp(t,A))){for(q=-1,p=-1,E=-1,S=-1,P=0,y=0;y<2;y++)for(r=t.D[y][A],b=0;b<t.e[r];b++)u=t.k[r][b],u!=A&&n.fm(t,u)==1&&(D=t.i[r][b],V=n.Jp(t,u,D),P<V&&(P=V,p=D,q=u,E=r,S=t.D[1-y][A]));if(p!=-1){for(T=0;T<2;T++)for(b=0;b<t.e[t.D[T][A]];b++)u=t.k[t.D[T][A]][b],u!=A&&n.fm(t,u)==1&&(t.H[u]=1);for(t.D[1][q]!=p&&(t.D[0][q]=t.D[1][q],t.D[1][q]=p),Q=qt,c=0;c<t.j[E];c++)D=t.i[E][c],t.k[E][c]!=A&&Q>D&&(Q=D);for(k=B(o.Q7,H,6,2,15,1),v=0,L=0;L<t.j[S];L++)t.k[S][L]!=A&&(k[v++]=t.i[S][L]);i=n.no(t.J[E].a,t.J[E].b,t.J[S].a,t.J[S].b),e=0,v==2?(k[0]>k[1]&&(G=k[0],k[0]=k[1],k[1]=G),l=n.oo(i,n.Yl(t,S,k[0])),g=n.oo(i,n.Yl(t,S,k[1])),e=l-g):e=n.oo(i,n.Yl(t,S,k[0])),e<0^(t.F[A]&3)==2^Q==p?t.H[q]=257:t.H[q]=129;}}},n.yq=function(t){var A,e,r,P;for(n.bw(t,7),r=0;r<t.g;r++)(t.H[r]==257||t.H[r]==129)&&(t.H[r]=1);for(A=0;A<t.f;A++)n.wq(t,A);for(P=0;P<t.g;P++)n.xq(t,P);for(e=0;e<t.g;e++)t.H[e]==2&&(t.F[e]&3)==3&&(t.H[e]=386);},n.zq=function(t,A){var e,r,P,i,D,u,Q,l,L,y;if(u=B(o.Q7,H,6,t.q,15,1),D=n.hp(t,u,false,A),D<=1)return null;for(Q=B(o.Q7,H,6,D,15,1),r=0;r<t.f;r++)++Q[u[r]];for(L=0,y=Q[0],l=1;l<D;l++)y<Q[l]&&(y=Q[l],L=l);for(e=0;e<t.q;e++)u[e]!=L&&(t.C[e]=-1);for(i=0;i<t.r;i++)(!A&&t.H[i]==16||u[t.D[0][i]]!=L)&&(t.H[i]=512);P=n.kl(t),t.T=0;try{n.zo(t,!0,!0);}catch(T){if(T=C1(T),!Zt(T,19))throw st(T);}return P;},n.Aq=function(t,A){return (t.u[A]&bA)!=0?true:t.C[A]<=1?false:n.Nm(t,A)||t.C[A]==13||t.C[A]>=171;},n.Bq=function(t){var A,e,r,P,i,D,u,Q,l,L;for(i=n.Wl(t,t.q,t.r,n.Qk),D=i*i/16,r=1;r<t.q;r++)for(P=0;P<r;P++)if(Q=t.J[P].a-t.J[r].a,l=t.J[P].b-t.J[r].b,L=t.J[P].c-t.J[r].c,Q*Q+l*l+L*L<D)throw st(new h.cC("The distance between two atoms is too close."));for(n.bw(t,1),A=0,e=0;e<t.f;e++){if(n.vp(t,e)>(u=n.pm(t,e),u+n.nm(t,e,u,t.C[e])))throw st(new h.cC("atom valence exceeded"));A+=t.s[e];}if(A!=0)throw st(new h.cC("unbalanced atom charge"));},n.Cq=function(t){var A,e;if(t.K)for(n.bw(t,7),A=0;A<t.f;A++)(t.u[A]&V0)!=0&&(S1(J(t.B[A],120),8)&&(t.B[A]=J(t.B[A],-9)),t.B[A]=J(t.B[A],-4294967297)),A<t.f&&n.ot(t.p,A)?t.B[A]=J(t.B[A],{l:4194297,m:W1,h:1048571}):pt(J(t.B[A],2),0)&&(t.B[A]=J(t.B[A],-5)),(pt(J(t.B[A],vA),0)||pt(J(t.B[A],EA),0)&&S1(J(t.B[A],lA),0))&&S1(J(t.B[A],120),8)&&(t.B[A]=J(t.B[A],-9)),t.s[A]!=0&&(t.B[A]=J(t.B[A],-234881025)),n.vp(t,A)==(e=n.pm(t,A),e+n.nm(t,A,e,t.C[A]))&&(t.B[A]=J(t.B[A],-4063233),t.B[A]=J(t.B[A],-17042430230529),t.B[A]=J(t.B[A],-1921),t.B[A]=J(t.B[A],-114689)),pt(J(t.B[A],i0),0)&&t.w[A]!=0&&n.an(t,t.w[A]);},n.Dq=function(t){var A,e,r;if(t.K)for(n.bw(t,7),A=0;A<t.g;A++)r=rn(t.G[A]&127),(A<t.g&&(n.ut(t.p,A)||t.H[A]==8))&(t.G[A]&8)!=0&&(t.G[A]&=-9,--r),r!=0&&(e=t.H[A]&127,e==1?t.G[A]|=1:e==2?t.G[A]|=2:e==4?t.G[A]|=4:e==32?t.G[A]|=32:e==64?t.G[A]|=64:e==16?t.G[A]|=16:e==8&&(t.G[A]|=8));},n.Eq=function(t,A,e){var r;return r=n.ao(t,A,e),r&&e==386&&(n.bw(t,7),r=r&(t.F[A]&128)==0),r;},n.Fq=function(){n.eo.call(this);},n.Gq=function(t,A){n.fo.call(this,t,A);},n.Hq=function(t){n.fo.call(this,t?t.M:256,t?t.N:256),t&&n.ol(t,this);},x(114,88,{114:1,88:1,4:1}),s.$b=function(t){n.Eo(this,t);},s.Zb=function(t){return n.Yp(this,t);},s.f=0,s.g=0,o.p8=N(114),n.Kq=function(t,A){return d.Math.pow(10,d.Math.log(2e3)*d.Math.LOG10E*t/(A-1)-1);},n.Lq=function(t,A){var e,r;for(e=A,r=0;A!=0;)t.d==0&&(t.f=(t.c[++t.e]&63)<<11,t.d=6),r|=(H1&t.f)>>16-e+A,t.f<<=1,--A,--t.d;return r;},n.Mq=function(t,A,e){t.d=6,t.e=e,t.c=A,t.f=(A[t.e]&63)<<11;},n.Nq=function(t,A){var e,r,P,i;return r=A/2|0,P=t>=r,P&&(t-=r),i=A/32,e=i*t/(r-t),P?-e:e;},n.Oq=function(t){var A,e,r,P;if(t.g.K)for(A=0;A<t.g.r;A++)r=n.hm(t.g,A),(r&127)!=0&&(e=-1,P=0,(r&16)!=0&&(e=16,++P),(r&64)!=0&&(e=64,++P),(r&32)!=0&&(e=32,++P),(r&4)!=0&&(e=4,++P),(r&2)!=0&&(e=2,++P),(r&8)!=0&&(e=8,++P),(r&1)!=0&&(e=1,++P),e!=-1&&(n.Mn(t.g,A,e),P==1&&n.Ln(t.g,A,127,false)));},n.Pq=function(t,A){var e;return A==null||w.SAb(A).length==0?null:n.Rq(t,w.sAb((e=A,q1.Cnb(),e)));},n.Qq=function(t,A,e){var r,P;return A==null?null:n.Sq(t,w.sAb((P=A,q1.Cnb(),P)),e==null?null:w.sAb((r=e,r)),0);},n.Rq=function(t,A){var e;if(A==null||A.length==0)return null;for(e=2;e<A.length-2;e++)if(A[e]==32)return n.Sq(t,A,A,e+1);return n.Sq(t,A,null,0);},n.Sq=function(t,A,e,r){var P,i,D,u,Q;return A==null?null:(n.Mq(t,A,0),P=n.Lq(t,4),u=n.Lq(t,4),P>8&&(P=u),i=n.Lq(t,P),D=n.Lq(t,u),Q=new n.uw(i,D),n.Vq(t,Q,A,e,r),Q);},n.Tq=function(t,A,e){var r,P,i,D;if(e==null||w.SAb(e).length==0){n.Uq(t,A,null,null);return;}r=h.hkb(e,X1(32)),r>0&&r<w.SAb(e).length-1?n.Uq(t,A,w.sAb((i=(w.MAb(0,r,w.SAb(e).length),w.SAb(e).substr(0,r)),q1.Cnb(),i)),w.sAb((D=(w.NAb(r+1,w.SAb(e).length+1),w.SAb(e).substr(r+1)),D))):n.Uq(t,A,w.sAb((P=e,q1.Cnb(),P)),null);},n.Uq=function(t,A,e,r){if(e==null||e.length==0){n.jl(A),A.d=null;return;}n.Vq(t,A,e,r,0);},n.Vq=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt,c1,D1,St,gt,w1,I1,L1,G1,s1,Q1,z1,t0,p0,J1,u1,u0,h0,d0,RA,BA,nA,eA,R0,CA,K0,C0,YA,dn,vt,Wt,kA,mn,vn,Fn,i8,f8,D8,u8,Q8,o8,l8,L8,qr,Vr,s8,U8,y8,T8,h8,c8,S0,b8,w8,g8,B8,k8,d8,m8,v8,F8,j8,QA,p8,Pn,M8,_n,ru,Pu,iu,fu,Du,$t,R8,uu,Qu,ou,XA,lu,oA,Wn,Ir,Re,Lu,rA,Or,su,Ce,Uu,Se,yu,He;if(n.jl(A),A.d=null,!(e==null||0>=e.length)){if(t.g=A,Or=8,r!=null&&(P<0||P>=r.length)&&(r=null),n.Mq(t,e,0),u=n.Lq(t,4),G=n.Lq(t,4),u>8&&(Or=u,u=G),u==0){n.On(t.g,n.Lq(t,1)==1);return;}for(Q=n.Lq(t,u),l=n.Lq(t,G),Du=n.Lq(t,u),ou=n.Lq(t,u),uu=n.Lq(t,u),Ut=n.Lq(t,u),c=0;c<Q;c++)n.Uk(t.g,6);for(h0=0;h0<Du;h0++)n.Cn(t.g,n.Lq(t,u),7);for(d0=0;d0<ou;d0++)n.Cn(t.g,n.Lq(t,u),8);for(vt=0;vt<uu;vt++)n.Cn(t.g,n.Lq(t,u),n.Lq(t,8));for(o8=0;o8<Ut;o8++)n.jn(t.g,n.Lq(t,u),n.Lq(t,4)-8);for(wt=1+l-Q,kt=n.Lq(t,4),V=0,n.zn(t.g,0,0),n.An(t.g,0,0),n.Bn(t.g,0,0),Bt=r!=null&&r[P]>=39,rA=0,Ce=0,Se=0,He=0,Ht=false,Kt=false,Bt&&(r.length>2*Q-2&&r[2*Q-2]==39||r.length>3*Q-3&&r[3*Q-3]==39?(Kt=true,Ht=r.length==3*Q-3+9,QA=Ht?3*Q-3:2*Q-2,q=86*(r[QA+1]-40)+r[QA+2]-40,rA=d.Math.pow(10,q/2e3-1),QA+=2,su=86*(r[QA+1]-40)+r[QA+2]-40,Ce=d.Math.pow(10,su/1500-1),QA+=2,Uu=86*(r[QA+1]-40)+r[QA+2]-40,Se=d.Math.pow(10,Uu/1500-1),Ht&&(QA+=2,yu=86*(r[QA+1]-40)+r[QA+2]-40,He=d.Math.pow(10,yu/1500-1))):Ht=r.length==3*Q-3),t._b()&&Ht&&(r=null,Bt=false),S0=1;S0<Q;S0++){if(St=n.Lq(t,kt),St==0){Bt&&(n.zn(t.g,S0,n.Ql(t.g,0)+8*(r[S0*2-2]-83)),n.An(t.g,S0,n.Rl(t.g,0)+8*(r[S0*2-1]-83)),Ht&&n.Bn(t.g,S0,n.Sl(t.g,0)+8*(r[2*Q-3+S0]-83))),++wt;continue;}V+=St-1,Bt&&(n.zn(t.g,S0,n.Ql(t.g,V)+r[S0*2-2]-83),n.An(t.g,S0,n.Rl(t.g,V)+r[S0*2-1]-83),Ht&&n.Bn(t.g,S0,n.Sl(t.g,V)+(r[2*Q-3+S0]-83))),n.Wk(t.g,V,S0,1);}for(d8=0;d8<wt;d8++)n.Wk(t.g,n.Lq(t,u),n.Lq(t,u),1);for(p8=B(o.vfb,Tt,6,l,16,1),it=0;it<l;it++)switch(lt=n.Lq(t,2),lt){case 0:p8[it]=true;break;case 2:n.Mn(t.g,it,2);break;case 3:n.Mn(t.g,it,4);}for(D=n.Lq(t,u),m8=0;m8<D;m8++)if(T=n.Lq(t,u),Or==8)XA=n.Lq(t,2),XA==3?(n.on(t.g,T,1,0),n.un(t.g,T,1,false)):n.un(t.g,T,XA,false);else switch(XA=n.Lq(t,3),XA){case 4:n.un(t.g,T,1,false),n.on(t.g,T,1,n.Lq(t,3));break;case 5:n.un(t.g,T,2,false),n.on(t.g,T,1,n.Lq(t,3));break;case 6:n.un(t.g,T,1,false),n.on(t.g,T,2,n.Lq(t,3));break;case 7:n.un(t.g,T,2,false),n.on(t.g,T,2,n.Lq(t,3));break;default:n.un(t.g,T,XA,false);}for(Or==8&&n.Lq(t,1)==0&&(t.g.L=true),i=n.Lq(t,G),v8=0;v8<i;v8++)if(O=n.Lq(t,G),n.im(t.g,O)==1)switch(XA=n.Lq(t,3),XA){case 4:n.Jn(t.g,O,1,false),n.Gn(t.g,O,1,n.Lq(t,3));break;case 5:n.Jn(t.g,O,2,false),n.Gn(t.g,O,1,n.Lq(t,3));break;case 6:n.Jn(t.g,O,1,false),n.Gn(t.g,O,2,n.Lq(t,3));break;case 7:n.Jn(t.g,O,2,false),n.Gn(t.g,O,2,n.Lq(t,3));break;default:n.Jn(t.g,O,XA,false);}else n.Jn(t.g,O,n.Lq(t,2),false);for(n.On(t.g,n.Lq(t,1)==1),y=null,Re=null,R8=0;n.Lq(t,1)==1;)switch(Nt=R8+n.Lq(t,4),Nt){case 0:for($t=n.Lq(t,u),F8=0;F8<$t;F8++)T=n.Lq(t,u),n.vn(t.g,T,B1,true);break;case 1:for($t=n.Lq(t,u),j8=0;j8<$t;j8++)T=n.Lq(t,u),iu=n.Lq(t,8),n.tn(t.g,T,iu);break;case 2:for($t=n.Lq(t,G),RA=0;RA<$t;RA++)n.Lq(t,G);break;case 3:for($t=n.Lq(t,u),BA=0;BA<$t;BA++)T=n.Lq(t,u),n.vn(t.g,T,Z1,true);break;case 4:for($t=n.Lq(t,u),nA=0;nA<$t;nA++)T=n.Lq(t,u),Ir=h1(Ct(n.Lq(t,4)),3),n.vn(t.g,T,Ir,true);break;case 5:for($t=n.Lq(t,u),eA=0;eA<$t;eA++)T=n.Lq(t,u),L=h1(Ct(n.Lq(t,2)),1),n.vn(t.g,T,L,true);break;case 6:for($t=n.Lq(t,u),R0=0;R0<$t;R0++)T=n.Lq(t,u),n.vn(t.g,T,1,true);break;case 7:for($t=n.Lq(t,u),CA=0;CA<$t;CA++)T=n.Lq(t,u),J1=h1(Ct(n.Lq(t,4)),7),n.vn(t.g,T,J1,true);break;case 8:for($t=n.Lq(t,u),K0=0;K0<$t;K0++){for(T=n.Lq(t,u),S=n.Lq(t,4),k=B(o.Q7,H,6,S,15,1),Pn=0;Pn<S;Pn++)v=n.Lq(t,8),k[Pn]=v;n.pn(t.g,T,k);}break;case 9:for($t=n.Lq(t,G),C0=0;C0<$t;C0++)O=n.Lq(t,G),Ir=n.Lq(t,2)<<7,n.Ln(t.g,O,Ir,true);break;case 10:for($t=n.Lq(t,G),YA=0;YA<$t;YA++)O=n.Lq(t,G),Dt=n.Lq(t,5),n.Ln(t.g,O,Dt,true);break;case 11:for($t=n.Lq(t,u),dn=0;dn<$t;dn++)T=n.Lq(t,u),n.vn(t.g,T,L0,true);break;case 12:for($t=n.Lq(t,G),Wt=0;Wt<$t;Wt++)O=n.Lq(t,G),ot=n.Lq(t,8)<<9,n.Ln(t.g,O,ot,true);break;case 13:for($t=n.Lq(t,u),kA=0;kA<$t;kA++)T=n.Lq(t,u),lu=h1(Ct(n.Lq(t,3)),14),n.vn(t.g,T,lu,true);break;case 14:for($t=n.Lq(t,u),mn=0;mn<$t;mn++)T=n.Lq(t,u),fu=h1(Ct(n.Lq(t,5)),17),n.vn(t.g,T,fu,true);break;case 15:case 31:R8+=16;break;case 16:for($t=n.Lq(t,u),vn=0;vn<$t;vn++)T=n.Lq(t,u),Wn=h1(Ct(n.Lq(t,3)),22),n.vn(t.g,T,Wn,true);break;case 17:for($t=n.Lq(t,u),Fn=0;Fn<$t;Fn++)T=n.Lq(t,u),n.gn(t.g,T,n.Lq(t,4));break;case 18:for($t=n.Lq(t,u),Pu=n.Lq(t,4),i8=0;i8<$t;i8++){for(T=n.Lq(t,u),xt=n.Lq(t,Pu),M8=B(o.M7,Y,6,xt,15,1),Pn=0;Pn<xt;Pn++)M8[Pn]=n.Lq(t,7)<<24>>24;n.mn(t.g,T,h.$jb((_n=M8,q1.Cnb(),_n)));}break;case 19:for($t=n.Lq(t,u),f8=0;f8<$t;f8++)T=n.Lq(t,u),bt=h1(Ct(n.Lq(t,3)),25),n.vn(t.g,T,bt,true);break;case 20:for($t=n.Lq(t,G),D8=0;D8<$t;D8++)O=n.Lq(t,G),Wn=n.Lq(t,3)<<17,n.Ln(t.g,O,Wn,true);break;case 21:for($t=n.Lq(t,u),u8=0;u8<$t;u8++)T=n.Lq(t,u),n.wn(t.g,T,n.Lq(t,2)<<4);break;case 22:for($t=n.Lq(t,u),Q8=0;Q8<$t;Q8++)T=n.Lq(t,u),n.vn(t.g,T,NA,true);break;case 23:for($t=n.Lq(t,G),l8=0;l8<$t;l8++)O=n.Lq(t,G),n.Ln(t.g,O,TA,true);break;case 24:for($t=n.Lq(t,G),L8=0;L8<$t;L8++)O=n.Lq(t,G),L=n.Lq(t,2)<<21,n.Ln(t.g,O,L,true);break;case 25:for(qr=0;qr<Q;qr++)n.Lq(t,1)==1&&n.xn(t.g,qr,true);break;case 26:for($t=n.Lq(t,G),y=B(o.Q7,H,6,$t,15,1),Vr=0;Vr<$t;Vr++)y[Vr]=n.Lq(t,G);break;case 27:for($t=n.Lq(t,u),s8=0;s8<$t;s8++)T=n.Lq(t,u),n.vn(t.g,T,i0,true);break;case 28:for($t=n.Lq(t,G),U8=0;U8<$t;U8++)n.Mn(t.g,n.Lq(t,G),16);break;case 29:for($t=n.Lq(t,u),y8=0;y8<$t;y8++)T=n.Lq(t,u),p0=h1(Ct(n.Lq(t,2)),30),n.vn(t.g,T,p0,true);break;case 30:for($t=n.Lq(t,u),T8=0;T8<$t;T8++)T=n.Lq(t,u),Wn=h1(Ct(n.Lq(t,7)),32),n.vn(t.g,T,Wn,true);break;case 32:for($t=n.Lq(t,u),h8=0;h8<$t;h8++)T=n.Lq(t,u),Lu=h1(Ct(n.Lq(t,2)),44),n.vn(t.g,T,Lu,true);break;case 33:for($t=n.Lq(t,u),c8=0;c8<$t;c8++)T=n.Lq(t,u),G1=h1(Ct(n.Lq(t,5)),39),n.vn(t.g,T,G1,true);break;case 34:for($t=n.Lq(t,u),b8=0;b8<$t;b8++)T=n.Lq(t,u),n.vn(t.g,T,ee,true);break;case 35:for($t=n.Lq(t,G),w8=0;w8<$t;w8++)O=n.Lq(t,G),n.Ln(t.g,O,sn,true);break;case 36:for($t=n.Lq(t,G),g8=0;g8<$t;g8++)O=n.Lq(t,G),ut=n.Lq(t,2)<<5,n.Ln(t.g,O,ut,true);break;case 37:for($t=n.Lq(t,G),B8=0;B8<$t;B8++)O=n.Lq(t,G),ut=n.Lq(t,1)==0?32:64,n.Mn(t.g,O,ut);break;case 38:for($t=n.Lq(t,u),ct=n.Lq(t,3),Re=B(o.Q7,H,6,Q,15,1),k8=0;k8<$t;k8++)T=n.Lq(t,u),It=n.Lq(t,ct),Re[T]=It;break;case 39:for($t=n.Lq(t,u),u0=0;u0<$t;u0++)T=n.Lq(t,u),Qu=h1(Ct(n.Lq(t,4)),47),n.vn(t.g,T,Qu,true);}if(n.Ge(new n.Ve(t.g),p8,false),y!=null)for(tt=y,Qt=0,z=tt.length;Qt<z;++Qt)O=tt[Qt],n.Mn(t.g,O,n.im(t.g,O)==2?4:2);if(r==null&&!t.i&&e.length>t.e+1&&(e[t.e+1]==32||e[t.e+1]==9)&&(r=e,P=t.e+2),r!=null)try{if(r[P]==33||r[P]==35){for(n.Mq(t,r,P+1),Ht=n.Lq(t,1)==1,Kt=n.Lq(t,1)==1,oA=2*n.Lq(t,4),K=1<<oA,z1=0,O=0,b=1;b<Q;b++)O<l&&n.Zl(t.g,1,O)==b?(z1=n.Zl(t.g,0,O++),Q1=1):(z1=0,Q1=8),c1=Q1*(n.Lq(t,oA)+1-(K>>1)),D1=Q1*(n.Lq(t,oA)+1-(K>>1)),n.zn(t.g,b,n.Ql(t.g,z1)+c1),n.An(t.g,b,n.Rl(t.g,z1)+D1),Ht&&n.Bn(t.g,b,n.Sl(t.g,z1)+Q1*(n.Lq(t,oA)+1-(K>>1)));if(r[P]==35){for(u1=0,t0=B(o.Q7,H,6,Q,15,1),g=0;g<Q;g++)u1+=t0[g]=n.op(t.g,g);for(T=0;T<Q;T++)for(u0=0;u0<t0[T];u0++)J1=n.Uk(t.g,1),n.Wk(t.g,T,J1,1),n.zn(t.g,J1,n.Ql(t.g,T)+(n.Lq(t,oA)+1-(K>>1))),n.An(t.g,J1,n.Rl(t.g,T)+(n.Lq(t,oA)+1-(K>>1))),Ht&&n.Bn(t.g,J1,n.Sl(t.g,T)+(n.Lq(t,oA)+1-(K>>1))),Re!=null&&(Re[T]&1<<u0)!=0&&n.xn(t.g,J1,!0);Q+=u1,l+=u1;}if(E=Ht?1.5:(n.Rk(),n.Rk(),n.Qk),p=n.Wl(t.g,Q,l,E),Kt)for(rA=n.Kq(n.Lq(t,oA),K),Ce=rA*n.Nq(n.Lq(t,oA),K),Se=rA*n.Nq(n.Lq(t,oA),K),Ht&&(He=rA*n.Nq(n.Lq(t,oA),K)),Q1=rA/p,T=0;T<Q;T++)n.zn(t.g,T,Ce+Q1*n.Ql(t.g,T)),n.An(t.g,T,Se+Q1*n.Rl(t.g,T)),Ht&&n.Bn(t.g,T,He+Q1*n.Sl(t.g,T));else for(rA=1.5,Q1=rA/p,T=0;T<Q;T++)n.zn(t.g,T,Q1*n.Ql(t.g,T)),n.An(t.g,T,Q1*n.Rl(t.g,T)),Ht&&n.Bn(t.g,T,Q1*n.Sl(t.g,T));}else if(Ht&&!Kt&&rA==0&&(rA=1.5),rA!=0&&t.g.r!=0){for(p=0,O=0;O<t.g.r;O++)gt=n.Ql(t.g,n.Zl(t.g,0,O))-n.Ql(t.g,n.Zl(t.g,1,O)),w1=n.Rl(t.g,n.Zl(t.g,0,O))-n.Rl(t.g,n.Zl(t.g,1,O)),I1=Ht?n.Sl(t.g,n.Zl(t.g,0,O))-n.Sl(t.g,n.Zl(t.g,1,O)):0,p+=d.Math.sqrt(gt*gt+w1*w1+I1*I1);for(p/=t.g.r,s1=rA/p,T=0;T<t.g.q;T++)n.zn(t.g,T,n.Ql(t.g,T)*s1+Ce),n.An(t.g,T,n.Rl(t.g,T)*s1+Se),Ht&&n.Bn(t.g,T,n.Sl(t.g,T)*s1+He);}}catch(dA){if(dA=C1(dA),Zt(dA,19))L1=dA,h.MB(L1,(h.Pkb(),h.Okb),""),""+L1+h.$jb((ru=e,q1.Cnb(),ru))+h.$jb((_n=r,_n)),r=null,Ht=false;else throw st(dA);}if(Et=r!=null&&!Ht,n.Oq(t),Et||t._b())for(n.bw(t.g,7),O=0;O<t.g.g;O++)n.fm(t.g,O)==2&&!n.jq(t.g,O)&&n.gm(t.g,O)==0&&n.Kn(t.g,O);if(n.vq(t.g,0),!Et&&t._b())try{t.ac(t.g),Et=!0;}catch(dA){if(dA=C1(dA),Zt(dA,19))L1=dA,h.MB(L1,(h.Pkb(),h.Okb),""),""+L1+h.$jb((_n=e,q1.Cnb(),_n));else throw st(dA);}Et?(n.yq(t.g),n.ow(t.g)):Ht||n.vq(t.g,0);}},n.Wq=function(t,A){var e,r,P,i,D,u;if(!(A.length<=0||A[0]<64))for(n.Mq(t,A,0),u=n.Lq(t,4),P=n.Lq(t,1)==1,i=n.Lq(t,1)==1,e=0;e<t.g.f;e++)D=n.Lq(t,u),r=P,P&&i&&(r=n.Lq(t,1)==1),n.rn(t.g,e,D,r);},n.Xq=function(){},x(319,1,{},n.Xq),s._b=function(){return  false;},s.ac=function(t){throw st(new h.cC("Unexpected request to invent coordinates. Check source code logic!"));},s.d=0,s.e=0,s.f=0,s.i=false,o.q8=N(319),n.$q=function(t){},n._q=function(t){n.$q(this),this.b=t;},x(50,319,{},n._q),s._b=function(){return this.b;},s.ac=function(t){var A;A=new X.CA(this.a),A.g=new U.uwb(78187493520),X.jA(A,t);},s.a=2,s.b=false,o.r8=N(50),n.cr=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v;P=A.a,c=A.g,l=pt(J(n.Ol(t.k,P),1),0),i=n.Jl(t.k,P),y=i!=null?n.jr(t,P,i,r):l?"*":n.Il(t.k,P),!l&&i==null&&n.Tp(t.k,P)&&(t.j&4)==0&&(n.Qo(t.k,P)!=0||n.zl(t.k,P)==-1&&n.Pl(t.k,P)==0)&&(y=w.SAb(y).toLowerCase()),A.f&&(e.a+="("),c!=-1&&n.gr(t,n.Xo(t.k,A.a,A.g),A.g,e),D=n.Bl(t.k,P),D==0&&(t.j&1)!=0&&(u=J(n.Ol(t.k,P),mA),nt(u,on)==0?D=-1:nt(u,Qn)==0&&(D=1)),L=n.Ml(t.k,P),T=(t.j&2)!=0?n.Ll(t.k,P):0,b=(t.j&1)!=0?n.pr(t,P,r):null,v=!l&&!n.wr(n.Tl(t.k,P))||i!=null||(n.Nl(t.k,P)==1||n.Nl(t.k,P)==2)&&(k=n.Hl(t.k,P)-1,!(k!=-1&&t.i[k][n.Gl(t.k,P)]<=1))&&(n.Tl(t.k,P)!=7||n.Bl(t.k,P)>0)||n.Tp(t.k,P)&&n.Qo(t.k,P)==0&&(t.j&4)==0||D!=0||L!=0||T!=0||n.zl(t.k,P)!=-1||n.Pl(t.k,P)!=0||b!=null,v&&(e.a+="["),L!=0&&(e.a+=L),e.a+=""+y,(n.Nl(t.k,P)==1||n.Nl(t.k,P)==2)&&(g=n.Hl(t.k,P)-1,!(g!=-1&&t.i[g][n.Gl(t.k,P)]<=1))&&(n.Tl(t.k,P)!=7||n.Bl(t.k,P)>0)&&h.Ikb(e,n.or(t,P,c)),(t.j&1)==0&&v&&(Q=n.Cp(t.k,P),Q==1?e.a+="H":Q>1&&(e.a+="H"+Q)),D!=0&&(e.a+=String.fromCharCode(D>0?43:45),d.Math.abs(D)>1&&h.Fkb(e,d.Math.abs(D))),b!=null&&(e.a+=""+b),T!=0&&(e.a+=":",e.a+=T),v&&(e.a+="]"),n.hr(t,A,e),A.e&&(e.a+=")");},n.dr=function(t,A,e){var r,P,i,D,u;for(i=true,u=U.xj(t.g,A).a,D=n.sr(t,u);D!=-1;)r=n.Zo(t.k,u,D),P=n._o(t.k,u,D),D=n.sr(t,r),n.fr(t,new n.$u(r,P,u,e&&i,e&&D==-1),++A),u=r,i=false;},n.er=function(t,A,e){var r,P,i,D;for(P=A,i=0,D=P.length;i<D;++i)r=P[i],t.f[r&1073741823]=(r&I0)!=0^e?2:1;},n.fr=function(t,A,e){U.rj(t.g,e,A),t.b[A.a]=true,A.g!=-1&&(t.c[n.Xo(t.k,A.a,A.g)]=true);},n.gr=function(t,A,e,r){var P,i,D,u,Q;Q=w.SAb(r.a).length,t.f[A]!=0&&h.Ekb(r,t.f[A]==1?47:92),t.j==1&&(P=n.hm(t.k,A)&127,P!=0&&((P&1)!=0&&t.f[A]==0&&(r.a+="-"),(P&2)!=0&&(w.SAb(r.a).length!=Q&&(r.a+=","),r.a+="="),(P&4)!=0&&(w.SAb(r.a).length!=Q&&(r.a+=","),r.a+="#"),(P&32)!=0&&(w.SAb(r.a).length!=Q&&(r.a+=","),r.a+="$"),(P&64)!=0&&(w.SAb(r.a).length!=Q&&(r.a+=","),r.a+="$"),(P&8)!=0&&(w.SAb(r.a).length!=Q&&(r.a+=","),r.a+=":"),(P&16)!=0&&(w.SAb(r.a).length!=Q&&(r.a+=","),h.Ikb(r,n.Lm(t.k,e)?"<-":"->")))),Q==w.SAb(r.a).length&&(!n.Up(t.k,A)||(t.j&4)!=0)&&(i=n.im(t.k,A)&127,i==1?n.Tp(t.k,n.Zl(t.k,0,A))&&n.Tp(t.k,n.Zl(t.k,1,A))&&(t.j&4)==0&&t.f[A]==0&&(r.a+="-"):i==2?r.a+="=":i==4?r.a+="#":i==32||i==64?r.a+="$":i==8?r.a+=":":i==16&&h.Ikb(r,n.Lm(t.k,e)?"<-":"->")),t.j==1&&(D=Q==w.SAb(r.a).length?"":";",u=n.hm(t.k,A)&384,u==256?r.a+=D+"@":u==128&&(r.a+=D+"!@"));},n.hr=function(t,A,e){var r,P,i;if(A.c!=null)for(P=0;P<A.c.length;P++)for(i=0;i<n.$o(t.k,A.a);i++)A.c[P]==n.Zo(t.k,A.a,i)&&(r=n._o(t.k,A.a,i),A.d[P]||n.gr(t,r,A.a,e),t.e[r]>9&&(e.a+="%"),h.Fkb(e,t.e[r]));},n.ir=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt;for(bt=new U.Mj(),v=new U.rrb(t.g);v.a<v.c.a.length;)if(k=U.qrb(v),k.g!=-1&&(p=n.Xo(t.k,k.a,k.g),!n.Vp(t.k,p)&&!n.jq(t.k,p)&&(n.gm(t.k,p)==1||n.gm(t.k,p)==2))){if(Dt=U.xj(t.g,t.q[k.g]),e=B(o.Q7,H,6,n.$o(t.k,k.a)+n.$o(t.k,Dt.a)-2,15,1),K=0,ot=false,Dt.g!=-1)e[K++]=Dt.b;else {for(E=-1,Ut=-1,q=qt,tt=0;tt<n.$o(t.k,Dt.a);tt++)y=n.Zo(t.k,Dt.a,tt),y!=k.a&&(E==-1?(E=tt,q=t.q[y]):q<t.q[y]?Ut=tt:(Ut=E,E=tt));Ut==-1?(it=n.Zo(t.k,Dt.a,E),lt=n._o(t.k,Dt.a,E),e[K++]=lt|(n.ur(t,Dt.a,it)?I0:0)):(T=n.Zo(t.k,Dt.a,E),b=n._o(t.k,Dt.a,E),c=n.Zo(t.k,Dt.a,Ut),g=n._o(t.k,Dt.a,Ut),e[K++]=b|(n.ur(t,Dt.a,T)?I0:0),e[K++]=g|(n.ur(t,Dt.a,c)?0:I0));}if(n.$o(t.k,Dt.a)==3&&Dt.g!=-1){for(tt=0;tt<n.$o(t.k,Dt.a);tt++)if(y=n.Zo(t.k,Dt.a,tt),y!=Dt.g&&y!=k.a){r=n._o(t.k,Dt.a,tt),e[K++]=r|(n.ur(t,Dt.a,y)?I0:0),y<Dt.g&&(ot=!ot);break;}}for(n.gm(t.k,p)==2&&(ot=!ot),O=0;O<n.$o(t.k,k.a);O++)if(Q=n.Zo(t.k,k.a,O),Q!=k.g){if(G=ot,n.$o(t.k,k.a)==3){for(z=0;z<n.$o(t.k,k.a);z++)if(y=n.Zo(t.k,k.a,z),y!=k.g&&y!=Q){y<Q&&(G=!G);break;}}n.Fm(t.k,p)&&(V=n.tg(t.d,p),t.n[V]||(t.n[V]=true,t.o[V]=G),t.o[V]&&(G=!G)),l=n.Xo(t.k,k.a,Q),e[K++]=l|(G^n.ur(t,k.a,Q)?0:I0);}w.iAb(bt.a,e);}for(t.f=B(o.Q7,H,6,t.k.g,15,1),bt.a.length!=0&&n.er(t,U.Ej(bt,0),false);bt.a.length!=0;){for(wt=bt.a.length,O=bt.a.length-1;O>=0;O--){for(e=(w.FAb(O,bt.a.length),bt.a[O]),ut=0,Qt=false,L=false,i=e,D=0,u=i.length;D<u;++D)P=i[D],A=P&1073741823,t.f[A]!=0&&(S=(P&I0)!=0^t.f[A]==2,ut==0?Qt=S:Qt!=S&&(L=true),++ut);ut!=0&&(e=U.Ej(bt,O),L||n.er(t,e,Qt));}wt==bt.a.length&&n.er(t,U.Ej(bt,0),false);}},n.jr=function(t,A,e,r){var P,i,D,u,Q,l,L;for(L=w.SAb(r.a).length,0<L?r.a=h.rkb(r.a,0,0):0>L&&(r.a+=h.xkb(B(o.N7,e0,6,-L,15,1))),Q=n.Tp(t.k,A)&&(t.j&4)==0,i=e,D=0,u=i.length;D<u;++D)P=i[D],w.SAb(r.a).length!=0&&(r.a+=","),l=(n.Rk(),n.Lk)[P],h.Ikb(r,Q?w.SAb(l).toLowerCase():l);return r.a;},n.kr=function(t){var A,e,r,P,i,D,u,Q;if(!t.k||t.k.q==0)return "";for(n.bw(t.k,15),t.d=new n.Og(t.k,129),P=t.d.T,t.o=B(o.vfb,Tt,6,P+1,16,1),t.n=B(o.vfb,Tt,6,P+1,16,1),t.i=U1(o.Q7,[E1,H],[7,6],15,[2,32],2),A=0;A<t.k.f;A++)Q=n.Hl(t.k,A)-1,Q!=-1&&++t.i[Q][n.Gl(t.k,A)];for(n.nr(t),n.lr(t),n.ir(t),r=new h.Kkb(),e=new h.Kkb(),i=true,u=new U.rrb(t.g);u.a<u.c.a.length;)D=U.qrb(u),D.g==-1&&(i?i=false:r.a+="."),n.cr(t,D,r,e);return r.a;},n.lr=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(y=new U.rrb(t.g);y.a<y.c.a.length;){for(l=U.qrb(y),e=0,i=0;i<n.$o(t.k,l.a);i++)t.c[n._o(t.k,l.a,i)]||++e;if(e!=0){for(l.c=B(o.Q7,H,6,e,15,1),e=0,D=0;D<n.$o(t.k,l.a);D++)t.c[n._o(t.k,l.a,D)]||(Q=n.Zo(t.k,l.a,D),l.c[e++]=t.q[Q]<<16|Q);for(w.lAb(l.c,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),P=0;P<l.c.length;P++)l.c[P]=e1&l.c[P];}}for(r=B(o.vfb,Tt,6,t.k.g,16,1),t.e=B(o.Q7,H,6,t.k.g,15,1),L=new U.rrb(t.g);L.a<L.c.a.length;)if(l=U.qrb(L),l.c!=null){for(l.d=B(o.vfb,Tt,6,l.c.length,16,1),P=0;P<l.c.length;P++)for(u=0;u<n.$o(t.k,l.a);u++)if(l.c[P]==n.Zo(t.k,l.a,u))if(A=n._o(t.k,l.a,u),t.c[A])r[t.e[A]]=false;else {for(t.c[A]=true,l.d[P]=true,t.e[A]=1;r[t.e[A]];)++t.e[A];r[t.e[A]]=true;}}},n.mr=function(t){var A,e,r,P;for(r=-1,P=qt,A=0;A<t.k.f;A++)t.b[A]||(e=t.a[A],n.Jm(t.k,A)&&(e+=I0),n.$o(t.k,A)==0?e+=1056964608:e+=n.$o(t.k,A)<<24,P>e&&(P=e,r=A));return r;},n.nr=function(t){var A,e,r,P,i;for(t.a=t.d.f,t.b=B(o.vfb,Tt,6,t.k.f,16,1),t.c=B(o.vfb,Tt,6,t.k.g,16,1),t.g=new U.Mj(),A=n.mr(t);A!=-1;){if(e=t.g.a.length,n.fr(t,new n.$u(A,-1,-1,false,false),e),n.$o(t.k,A)!=0)for(n.dr(t,e,false);e<t.g.a.length-1;){for(;n.tr(t,U.xj(t.g,e).a);)n.dr(t,e,true);++e;}A=n.mr(t);}for(t.q=B(o.Q7,H,6,t.k.f,15,1),r=0,i=new U.rrb(t.g);i.a<i.c.a.length;)P=U.qrb(i),t.q[P.a]=r++;},n.or=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;if(Q=false,n.Qo(t.k,A)!=0&&n.$o(t.k,A)==2&&n.ap(t.k,A,0)==2&&n.ap(t.k,A,1)==2){for(D=0;D<n.$o(t.k,A);D++)if(r=n.Zo(t.k,A,D),r!=e&&n.$o(t.k,r)==2){Q=true;break;}for(i=0;i<n.$o(t.k,A);i++){for(r=n.Zo(t.k,A,i),b=0,c=B(o.Q7,H,6,3,15,1),L=0;L<n.$o(t.k,r);L++)c[b]=n.Zo(t.k,r,L),c[b]!=A&&++b;b==2&&t.q[c[0]]<t.q[c[1]]^c[0]<c[1]&&(Q=!Q);}}else {for(y=B(o.Q7,H,6,4,15,1),T=B(o.Q7,H,6,4,15,1),u=0,e!=-1&&(y[u]=e,T[u++]=8*t.q[e]),(n.op(t.k,A)!=0||n.$o(t.k,A)==3)&&(y[u]=qt,T[u++]=8*t.q[A]),i=0;i<n.$o(t.k,A);i++)r=n.Zo(t.k,A,i),r!=e&&(y[u]=r,T[u++]=n.rr(t,A,i));Q=n.vr(y,T);}return l=n.Nl(t.k,A)==1^Q,n.ym(t.k,A)&&(P=n.ug(t.d,A),t.n[P]||(t.n[P]=true,t.o[P]=l),t.o[P]&&(l=!l)),l?"@@":"@";},n.pr=function(t,A,e){var r,P,i,D,u,Q,l,L;switch(u=w.SAb(e.a).length,0<u?e.a=h.rkb(e.a,0,0):0>u&&(e.a+=h.xkb(B(o.N7,e0,6,-u,15,1))),Q=n.Ol(t.k,A),P=Jt(A0(J(Q,mA),3)),P){case 20971520:e.a+="+0";break;case 25165824:n.Bl(t.k,A)==0&&(e.a+="-");break;case 12582912:n.Bl(t.k,A)==0&&(e.a+="+");}return r=J(Q,Xe),nt(r,2)==0?e.a+=";a":nt(r,4)==0&&(e.a+=";A"),i=J(Q,X0),nt(i,0)!=0&&(nt(i,1792)==0?e.a+=";H0":nt(i,1664)==0?e.a+=";H1":nt(i,1408)==0?e.a+=";H2":nt(i,896)==0?e.a+=";H3":nt(i,128)==0?e.a+=";!H0":nt(i,384)==0?e.a+=";!H0;!H1":nt(i,1536)==0?e.a+=";!H2;!H3":nt(i,f0)==0&&(e.a+=";!H3")),L=J(Q,120),nt(L,8)==0?e.a+=";!R0":nt(L,16)==0?e.a+=";!R1":nt(L,32)==0?e.a+=";!R2":nt(L,64)==0?e.a+=";!R3":nt(L,112)==0?e.a+=";R0":nt(L,104)==0?e.a+=";R1":nt(L,88)==0?e.a+=";R2":nt(L,56)==0&&(e.a+=";R3"),l=J(Q,EA),nt(l,lA)==0?e.a+=";!r"+U0(l):nt(l,K8)==0?e.a+=";r"+U0(l):nt(l,0)!=0&&(pt(J(l,un),0)?(S1(J(l,lA),0)&&(e.a+=";!r0"+U0(l)),S1(J(l,qA),0)&&(e.a+=";!r3"+U0(l)),S1(J(l,VA),0)&&(e.a+=";!r4"+U0(l)),S1(J(l,IA),0)&&(e.a+=";!r5"+U0(l)),S1(J(l,OA),0)&&(e.a+=";!r6"+U0(l)),S1(J(l,xA),0)&&(e.a+=";!r7"+U0(l))):(e.a+=";",pt(J(l,lA),0)&&(e.a+="r0,"+U0(l)),pt(J(l,qA),0)&&(e.a+="r3,"+U0(l)),pt(J(l,VA),0)&&(e.a+="r4,"+U0(l)),pt(J(l,IA),0)&&(e.a+="r5,"+U0(l)),pt(J(l,OA),0)&&(e.a+="r6,"+U0(l)),pt(J(l,xA),0)&&(e.a+="r7,"+U0(l)),h.whb(e,w.SAb(e.a).length-1))),nt(l,0)==0&&(l=A0(J(Q,vA),22),nt(l,0)!=0&&(e.a+=";r"+U0(l))),D=J(Q,aA),nt(D,Pe)==0&&(e.a+=";D1"),nt(D,Mn)==0&&(e.a+=";D2"),nt(D,Rn)==0&&(e.a+=";D3"),nt(D,ie)==0&&(e.a+=";!D3;!D4"),nt(D,j0)==0&&(e.a+=";!D4"),nt(D,tn)==0&&(e.a+=";!D0;!D1"),nt(D,iA)==0&&(e.a+=";!D0;!D1;!D2"),nt(D,ln)==0&&(e.a+=";!D0;!D1;!D2;!D3"),pt(J(Q,B1),0)&&h.Ikb(e,";D"+n.$o(t.k,A)),pt(J(Q,Z1),0)&&h.Ikb(e,";!D"+n.$o(t.k,A)),w.SAb(e.a).length==0?null:e.a;},n.qr=function(t){return t.p==null&&(t.p=n.kr(t)),t.p;},n.rr=function(t,A,e){var r,P,i,D,u;if(r=n._o(t.k,A,e),D=n.Zo(t.k,A,e),t.e[r]!=0){for(u=8*t.q[A]+1,P=U.xj(t.g,t.q[A]).c,i=0;i<P.length&&D!=P[i];i++)++u;return u;}return 8*t.q[D];},n.sr=function(t,A){var e,r,P,i,D;for(e=-1,r=-1,i=0;i<n.$o(t.k,A);i++)P=n.Zo(t.k,A,i),D=(n.ap(t.k,A,i)<<24)+t.a[P],!t.b[P]&&(e==-1||r<D)&&(e=i,r=D);return e;},n.tr=function(t,A){var e;for(e=0;e<n.$o(t.k,A);e++)if(!t.b[n.Zo(t.k,A,e)])return  true;return  false;},n.ur=function(t,A,e){var r,P;return r=U.xj(t.g,t.q[A]),r.g==e?false:(P=U.xj(t.g,t.q[e]),P.g==A?true:n.Zu(P,A));},n.vr=function(t,A){var e,r,P;for(r=false,e=1;e<4;e++)for(P=0;P<e;P++)t[P]>t[e]&&(r=!r),A[P]>A[e]&&(r=!r);return r;},n.wr=function(t){return t>=5&&t<=9||t>=15&&t<=17||t==35||t==53;},n.xr=function(t){n.yr.call(this,t,0);},n.yr=function(t,A){this.k=t,this.j=A;},n.zr=function(t){var A,e,r,P;for(P=new h.Kkb(),e=0;e<t.g.a.length;e++)e!=0&&(P.a+="."),h.Ikb(P,n.qr(new n.yr(U.xj(t.g,e),2)));for(P.a+=">",r=0;r<t.a.a.length;r++)r!=0&&(P.a+="."),h.Ikb(P,n.qr(new n.xr(U.xj(t.a,r))));for(P.a+=">",A=0;A<t.f.a.length;A++)A!=0&&(P.a+="."),h.Ikb(P,n.qr(new n.yr(U.xj(t.f,A),2)));return P.a;},x(111,1,{},n.xr,n.yr),s.j=0,o.s8=N(111),n.Ar=function(t,A){this.b=t,this.a=A;},x(3,1,{3:1},n.Ar),s.a=0,s.b=0,o.t8=N(3),n.Cr=function(){n.Cr=f1,n.Br=j(F(o.t8,2),b0,17,0,[null,j(F(o.t8,1),Vt,3,0,[new n.Ar(0,1.007825032),new n.Ar(1,2.014101778),new n.Ar(2,3.016049268),new n.Ar(3,4.027834627),new n.Ar(4,5.039542911),new n.Ar(5,6.044942608)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(1,3.01602931),new n.Ar(2,4.00260325),new n.Ar(3,5.012223628),new n.Ar(4,6.018888072),new n.Ar(5,7.028030527),new n.Ar(6,8.033921838),new n.Ar(7,9.043820323),new n.Ar(8,10.052399713)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(1,4.027182329),new n.Ar(2,5.012537796),new n.Ar(3,6.015122281),new n.Ar(4,7.016004049),new n.Ar(5,8.02248667),new n.Ar(6,9.026789122),new n.Ar(7,10.035480884),new n.Ar(8,11.043796166),new n.Ar(9,12.05378)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(1,5.04079),new n.Ar(2,6.019725804),new n.Ar(3,7.016929246),new n.Ar(4,8.005305094),new n.Ar(5,9.012182135),new n.Ar(6,10.01353372),new n.Ar(7,11.021657653),new n.Ar(8,12.026920631),new n.Ar(9,13.036133834),new n.Ar(10,14.042815522)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(2,7.029917389),new n.Ar(3,8.024606713),new n.Ar(4,9.013328806),new n.Ar(5,10.012937027),new n.Ar(6,11.009305466),new n.Ar(7,12.014352109),new n.Ar(8,13.017780267),new n.Ar(9,14.025404064),new n.Ar(10,15.031097291),new n.Ar(11,16.039808836),new n.Ar(12,17.046931399),new n.Ar(13,18.05617),new n.Ar(14,19.06373)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(2,8.037675026),new n.Ar(3,9.031040087),new n.Ar(4,10.01685311),new n.Ar(5,11.011433818),new n.Ar(6,12),new n.Ar(7,13.003354838),new n.Ar(8,14.003241988),new n.Ar(9,15.010599258),new n.Ar(10,16.014701243),new n.Ar(11,17.022583712),new n.Ar(12,18.026757058),new n.Ar(13,19.035248094),new n.Ar(14,20.040322395),new n.Ar(15,21.04934),new n.Ar(16,22.05645)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(3,10.042618),new n.Ar(4,11.026796226),new n.Ar(5,12.018613202),new n.Ar(6,13.005738584),new n.Ar(7,14.003074005),new n.Ar(8,15.000108898),new n.Ar(9,16.006101417),new n.Ar(10,17.008449673),new n.Ar(11,18.014081827),new n.Ar(12,19.017026896),new n.Ar(13,20.023367295),new n.Ar(14,21.027087574),new n.Ar(15,22.034440259),new n.Ar(16,23.04051),new n.Ar(17,24.0505)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(4,12.034404776),new n.Ar(5,13.0248104),new n.Ar(6,14.008595285),new n.Ar(7,15.003065386),new n.Ar(8,15.994914622),new n.Ar(9,16.999131501),new n.Ar(10,17.999160419),new n.Ar(11,19.00357873),new n.Ar(12,20.00407615),new n.Ar(13,21.008654631),new n.Ar(14,22.009967157),new n.Ar(15,23.015691325),new n.Ar(16,24.020369922),new n.Ar(17,25.02914),new n.Ar(18,26.03775)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(5,14.03608),new n.Ar(6,15.018010856),new n.Ar(7,16.01146573),new n.Ar(8,17.002095238),new n.Ar(9,18.000937667),new n.Ar(10,18.998403205),new n.Ar(11,19.999981324),new n.Ar(12,20.999948921),new n.Ar(13,22.00299925),new n.Ar(14,23.003574385),new n.Ar(15,24.008099371),new n.Ar(16,25.012094963),new n.Ar(17,26.019633157),new n.Ar(18,27.026892316),new n.Ar(19,28.03567),new n.Ar(20,29.04326)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(6,16.025756907),new n.Ar(7,17.017697565),new n.Ar(8,18.005697066),new n.Ar(9,19.001879839),new n.Ar(10,19.992440176),new n.Ar(11,20.993846744),new n.Ar(12,21.99138551),new n.Ar(13,22.994467337),new n.Ar(14,23.993615074),new n.Ar(15,24.997789899),new n.Ar(16,26.000461498),new n.Ar(17,27.0076152),new n.Ar(18,28.012108072),new n.Ar(19,29.019345902),new n.Ar(20,30.023872),new n.Ar(21,31.03311),new n.Ar(22,32.03991)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(7,18.02718),new n.Ar(8,19.01387945),new n.Ar(9,20.00734826),new n.Ar(10,20.997655099),new n.Ar(11,21.994436782),new n.Ar(12,22.989769675),new n.Ar(13,23.990963332),new n.Ar(14,24.989954352),new n.Ar(15,25.992589898),new n.Ar(16,26.994008702),new n.Ar(17,27.99889041),new n.Ar(18,29.002811301),new n.Ar(19,30.009226487),new n.Ar(20,31.013595108),new n.Ar(21,32.019649792),new n.Ar(22,33.027386),new n.Ar(23,34.0349),new n.Ar(24,35.04418)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(8,20.018862744),new n.Ar(9,21.011714174),new n.Ar(10,21.999574055),new n.Ar(11,22.99412485),new n.Ar(12,23.985041898),new n.Ar(13,24.985837023),new n.Ar(14,25.98259304),new n.Ar(15,26.984340742),new n.Ar(16,27.983876703),new n.Ar(17,28.988554743),new n.Ar(18,29.990464529),new n.Ar(19,30.996548459),new n.Ar(20,31.999145889),new n.Ar(21,33.005586975),new n.Ar(22,34.00907244),new n.Ar(23,35.018669),new n.Ar(24,36.02245),new n.Ar(25,37.03124)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(8,21.02804),new n.Ar(9,22.01952),new n.Ar(10,23.0072649),new n.Ar(11,23.999940911),new n.Ar(12,24.990428555),new n.Ar(13,25.986891659),new n.Ar(14,26.981538441),new n.Ar(15,27.981910184),new n.Ar(16,28.980444848),new n.Ar(17,29.982960304),new n.Ar(18,30.983946023),new n.Ar(19,31.988124379),new n.Ar(20,32.990869587),new n.Ar(21,33.996927255),new n.Ar(22,34.99993765),new n.Ar(23,36.006351501),new n.Ar(24,37.01031),new n.Ar(25,38.0169),new n.Ar(26,39.0219)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(8,22.03453),new n.Ar(9,23.02552),new n.Ar(10,24.011545711),new n.Ar(11,25.00410664),new n.Ar(12,25.992329935),new n.Ar(13,26.986704764),new n.Ar(14,27.976926533),new n.Ar(15,28.976494719),new n.Ar(16,29.973770218),new n.Ar(17,30.975363275),new n.Ar(18,31.974148129),new n.Ar(19,32.97800052),new n.Ar(20,33.978575745),new n.Ar(21,34.984584158),new n.Ar(22,35.986687363),new n.Ar(23,36.99299599),new n.Ar(24,37.99598),new n.Ar(25,39.0023),new n.Ar(26,40.0058),new n.Ar(27,41.0127),new n.Ar(28,42.0161)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(9,24.03435),new n.Ar(10,25.02026),new n.Ar(11,26.01178),new n.Ar(12,26.999191645),new n.Ar(13,27.99231233),new n.Ar(14,28.981801376),new n.Ar(15,29.978313807),new n.Ar(16,30.973761512),new n.Ar(17,31.973907163),new n.Ar(18,32.971725281),new n.Ar(19,33.973636381),new n.Ar(20,34.973314249),new n.Ar(21,35.978259824),new n.Ar(22,36.979608338),new n.Ar(23,37.98447),new n.Ar(24,38.98642),new n.Ar(25,39.99105),new n.Ar(26,40.9948),new n.Ar(27,42.00009),new n.Ar(28,43.00331),new n.Ar(29,44.00988),new n.Ar(30,45.01514),new n.Ar(31,46.02383)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(10,26.02788),new n.Ar(11,27.018795),new n.Ar(12,28.004372661),new n.Ar(13,28.996608805),new n.Ar(14,29.984902954),new n.Ar(15,30.979554421),new n.Ar(16,31.97207069),new n.Ar(17,32.971458497),new n.Ar(18,33.967866831),new n.Ar(19,34.96903214),new n.Ar(20,35.96708088),new n.Ar(21,36.971125716),new n.Ar(22,37.971163443),new n.Ar(23,38.975135275),new n.Ar(24,39.97547),new n.Ar(25,40.98003),new n.Ar(26,41.98149),new n.Ar(27,42.9866),new n.Ar(28,43.98832),new n.Ar(29,44.99482),new n.Ar(30,45.99957),new n.Ar(31,47.00762),new n.Ar(32,48.01299),new n.Ar(33,49.02201)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(11,28.02851),new n.Ar(12,29.01411),new n.Ar(13,30.00477),new n.Ar(14,30.992416014),new n.Ar(15,31.985688908),new n.Ar(16,32.977451798),new n.Ar(17,33.973761967),new n.Ar(18,34.968852707),new n.Ar(19,35.968306945),new n.Ar(20,36.9659026),new n.Ar(21,37.96801055),new n.Ar(22,38.968007677),new n.Ar(23,39.970415555),new n.Ar(24,40.970650212),new n.Ar(25,41.973174994),new n.Ar(26,42.974203385),new n.Ar(27,43.978538712),new n.Ar(28,44.9797),new n.Ar(29,45.98412),new n.Ar(30,46.98795),new n.Ar(31,47.99485),new n.Ar(32,48.99989),new n.Ar(33,50.00773),new n.Ar(34,51.01353)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(12,30.02156),new n.Ar(13,31.012126),new n.Ar(14,31.99766066),new n.Ar(15,32.989928719),new n.Ar(16,33.980270118),new n.Ar(17,34.975256726),new n.Ar(18,35.967546282),new n.Ar(19,36.966775912),new n.Ar(20,37.962732161),new n.Ar(21,38.964313413),new n.Ar(22,39.962383123),new n.Ar(23,40.964500828),new n.Ar(24,41.963046386),new n.Ar(25,42.965670701),new n.Ar(26,43.965365269),new n.Ar(27,44.968094979),new n.Ar(28,45.968093467),new n.Ar(29,46.972186238),new n.Ar(30,47.97507),new n.Ar(31,48.98218),new n.Ar(32,49.98594),new n.Ar(33,50.99324),new n.Ar(34,51.99817),new n.Ar(35,53.006227)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(13,32.02192),new n.Ar(14,33.00726),new n.Ar(15,33.99841),new n.Ar(16,34.988011615),new n.Ar(17,35.981293405),new n.Ar(18,36.973376915),new n.Ar(19,37.969080107),new n.Ar(20,38.963706861),new n.Ar(21,39.963998672),new n.Ar(22,40.961825972),new n.Ar(23,41.962403059),new n.Ar(24,42.960715746),new n.Ar(25,43.961556146),new n.Ar(26,44.960699658),new n.Ar(27,45.961976203),new n.Ar(28,46.961677807),new n.Ar(29,47.965512946),new n.Ar(30,48.967450084),new n.Ar(31,49.972782832),new n.Ar(32,50.97638),new n.Ar(33,51.98261),new n.Ar(34,52.98712),new n.Ar(35,53.99399),new n.Ar(36,54.999388)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(14,34.01412),new n.Ar(15,35.004765),new n.Ar(16,35.993087234),new n.Ar(17,36.985871505),new n.Ar(18,37.976318637),new n.Ar(19,38.970717729),new n.Ar(20,39.962591155),new n.Ar(21,40.962278349),new n.Ar(22,41.958618337),new n.Ar(23,42.958766833),new n.Ar(24,43.955481094),new n.Ar(25,44.956185938),new n.Ar(26,45.953692759),new n.Ar(27,46.954546459),new n.Ar(28,47.952533512),new n.Ar(29,48.955673302),new n.Ar(30,49.957518286),new n.Ar(31,50.961474238),new n.Ar(32,51.9651),new n.Ar(33,52.97005),new n.Ar(34,53.97468),new n.Ar(35,54.98055),new n.Ar(36,55.98579),new n.Ar(37,56.992356)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(15,36.01492),new n.Ar(16,37.00305),new n.Ar(17,37.9947),new n.Ar(18,38.984790009),new n.Ar(19,39.977964014),new n.Ar(20,40.969251316),new n.Ar(21,41.965516761),new n.Ar(22,42.96115098),new n.Ar(23,43.959403048),new n.Ar(24,44.955910243),new n.Ar(25,45.95517025),new n.Ar(26,46.952408027),new n.Ar(27,47.952234991),new n.Ar(28,48.950024065),new n.Ar(29,49.952187008),new n.Ar(30,50.9536027),new n.Ar(31,51.95665),new n.Ar(32,52.95817),new n.Ar(33,53.963),new n.Ar(34,54.9694),new n.Ar(35,55.97266),new n.Ar(36,56.97704),new n.Ar(37,57.98307),new n.Ar(38,58.988041)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(16,38.00977),new n.Ar(17,39.001323),new n.Ar(18,39.990498907),new n.Ar(19,40.983131),new n.Ar(20,41.973031622),new n.Ar(21,42.968523342),new n.Ar(22,43.959690235),new n.Ar(23,44.958124349),new n.Ar(24,45.952629491),new n.Ar(25,46.951763792),new n.Ar(26,47.947947053),new n.Ar(27,48.947870789),new n.Ar(28,49.944792069),new n.Ar(29,50.946616017),new n.Ar(30,51.946898175),new n.Ar(31,52.949731709),new n.Ar(32,53.95087),new n.Ar(33,54.95512),new n.Ar(34,55.95799),new n.Ar(35,56.9643),new n.Ar(36,57.96611),new n.Ar(37,58.97196),new n.Ar(38,59.97564),new n.Ar(39,60.982018)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(17,40.01109),new n.Ar(18,40.99974),new n.Ar(19,41.99123),new n.Ar(20,42.98065),new n.Ar(21,43.9744),new n.Ar(22,44.965782286),new n.Ar(23,45.960199491),new n.Ar(24,46.954906918),new n.Ar(25,47.95225448),new n.Ar(26,48.948516914),new n.Ar(27,49.947162792),new n.Ar(28,50.943963675),new n.Ar(29,51.944779658),new n.Ar(30,52.944342517),new n.Ar(31,53.946444381),new n.Ar(32,54.947238194),new n.Ar(33,55.95036),new n.Ar(34,56.95236),new n.Ar(35,57.95665),new n.Ar(36,58.9593),new n.Ar(37,59.9645),new n.Ar(38,60.96741),new n.Ar(39,61.97314),new n.Ar(40,62.97675)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(18,42.00643),new n.Ar(19,42.997707),new n.Ar(20,43.98547),new n.Ar(21,44.97916),new n.Ar(22,45.968361649),new n.Ar(23,46.962906512),new n.Ar(24,47.954035861),new n.Ar(25,48.951341135),new n.Ar(26,49.946049607),new n.Ar(27,50.944771767),new n.Ar(28,51.940511904),new n.Ar(29,52.940653781),new n.Ar(30,53.938884921),new n.Ar(31,54.940844164),new n.Ar(32,55.940645238),new n.Ar(33,56.9437538),new n.Ar(34,57.94425),new n.Ar(35,58.94863),new n.Ar(36,59.94973),new n.Ar(37,60.95409),new n.Ar(38,61.9558),new n.Ar(39,62.96186),new n.Ar(40,63.9642),new n.Ar(41,64.97037)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(19,44.00687),new n.Ar(20,44.99451),new n.Ar(21,45.98672),new n.Ar(22,46.9761),new n.Ar(23,47.96887),new n.Ar(24,48.959623415),new n.Ar(25,49.95424396),new n.Ar(26,50.948215487),new n.Ar(27,51.945570079),new n.Ar(28,52.941294702),new n.Ar(29,53.940363247),new n.Ar(30,54.938049636),new n.Ar(31,55.938909366),new n.Ar(32,56.938287458),new n.Ar(33,57.939986451),new n.Ar(34,58.940447166),new n.Ar(35,59.943193998),new n.Ar(36,60.94446),new n.Ar(37,61.94797),new n.Ar(38,62.94981),new n.Ar(39,63.95373),new n.Ar(40,64.9561),new n.Ar(41,65.96082),new n.Ar(42,66.96382)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(19,45.01456),new n.Ar(20,46.00081),new n.Ar(21,46.99289),new n.Ar(22,47.98056),new n.Ar(23,48.97361),new n.Ar(24,49.962993316),new n.Ar(25,50.956824936),new n.Ar(26,51.948116526),new n.Ar(27,52.945312282),new n.Ar(28,53.939614836),new n.Ar(29,54.938298029),new n.Ar(30,55.934942133),new n.Ar(31,56.935398707),new n.Ar(32,57.933280458),new n.Ar(33,58.934880493),new n.Ar(34,59.934076943),new n.Ar(35,60.936749461),new n.Ar(36,61.936770495),new n.Ar(37,62.940118442),new n.Ar(38,63.94087),new n.Ar(39,64.94494),new n.Ar(40,65.94598),new n.Ar(41,66.95),new n.Ar(42,67.95251),new n.Ar(43,68.9577)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(21,48.00176),new n.Ar(22,48.98972),new n.Ar(23,49.98154),new n.Ar(24,50.97072),new n.Ar(25,51.96359),new n.Ar(26,52.954224985),new n.Ar(27,53.948464147),new n.Ar(28,54.942003149),new n.Ar(29,55.939843937),new n.Ar(30,56.936296235),new n.Ar(31,57.935757571),new n.Ar(32,58.933200194),new n.Ar(33,59.933822196),new n.Ar(34,60.932479381),new n.Ar(35,61.934054212),new n.Ar(36,62.933615218),new n.Ar(37,63.935813523),new n.Ar(38,64.936484581),new n.Ar(39,65.939825412),new n.Ar(40,66.94061),new n.Ar(41,67.94436),new n.Ar(42,68.9452),new n.Ar(43,69.94981),new n.Ar(44,70.95173),new n.Ar(45,71.95641)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(22,49.99593),new n.Ar(23,50.98772),new n.Ar(24,51.97568),new n.Ar(25,52.96846),new n.Ar(26,53.957910508),new n.Ar(27,54.951336329),new n.Ar(28,55.942136339),new n.Ar(29,56.939800489),new n.Ar(30,57.935347922),new n.Ar(31,58.934351553),new n.Ar(32,59.930790633),new n.Ar(33,60.931060442),new n.Ar(34,61.928348763),new n.Ar(35,62.929672948),new n.Ar(36,63.927969574),new n.Ar(37,64.930088013),new n.Ar(38,65.929115232),new n.Ar(39,66.931569638),new n.Ar(40,67.931844932),new n.Ar(41,68.935181837),new n.Ar(42,69.93614),new n.Ar(43,70.94),new n.Ar(44,71.9413),new n.Ar(45,72.94608),new n.Ar(46,73.94791),new n.Ar(47,74.95297),new n.Ar(48,75.95533),new n.Ar(49,76.96083),new n.Ar(50,77.9638)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(23,51.99718),new n.Ar(24,52.98555),new n.Ar(25,53.97671),new n.Ar(26,54.96605),new n.Ar(27,55.95856),new n.Ar(28,56.949215695),new n.Ar(29,57.944540734),new n.Ar(30,58.939504114),new n.Ar(31,59.937368123),new n.Ar(32,60.933462181),new n.Ar(33,61.932587299),new n.Ar(34,62.929601079),new n.Ar(35,63.929767865),new n.Ar(36,64.927793707),new n.Ar(37,65.928873041),new n.Ar(38,66.927750294),new n.Ar(39,67.929637875),new n.Ar(40,68.929425281),new n.Ar(41,69.932409287),new n.Ar(42,70.932619818),new n.Ar(43,71.93552),new n.Ar(44,72.93649),new n.Ar(45,73.9402),new n.Ar(46,74.9417),new n.Ar(47,75.94599),new n.Ar(48,76.94795),new n.Ar(49,77.95281),new n.Ar(50,78.95528),new n.Ar(51,79.96189)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(24,53.99295),new n.Ar(25,54.98398),new n.Ar(26,55.97238),new n.Ar(27,56.96491),new n.Ar(28,57.954596465),new n.Ar(29,58.949267074),new n.Ar(30,59.941832031),new n.Ar(31,60.939513907),new n.Ar(32,61.934334132),new n.Ar(33,62.933215563),new n.Ar(34,63.929146578),new n.Ar(35,64.929245079),new n.Ar(36,65.926036763),new n.Ar(37,66.927130859),new n.Ar(38,67.924847566),new n.Ar(39,68.926553538),new n.Ar(40,69.92532487),new n.Ar(41,70.927727195),new n.Ar(42,71.926861122),new n.Ar(43,72.929779469),new n.Ar(44,73.929458261),new n.Ar(45,74.932937379),new n.Ar(46,75.933394207),new n.Ar(47,76.937085857),new n.Ar(48,77.938569576),new n.Ar(49,78.942095175),new n.Ar(50,79.944414722),new n.Ar(51,80.95048),new n.Ar(52,81.95484)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(25,55.99491),new n.Ar(26,56.98293),new n.Ar(27,57.97425),new n.Ar(28,58.96337),new n.Ar(29,59.95706),new n.Ar(30,60.94917),new n.Ar(31,61.944179608),new n.Ar(32,62.939141527),new n.Ar(33,63.936838307),new n.Ar(34,64.932739322),new n.Ar(35,65.931592355),new n.Ar(36,66.928204915),new n.Ar(37,67.927983497),new n.Ar(38,68.925580912),new n.Ar(39,69.926027741),new n.Ar(40,70.92470501),new n.Ar(41,71.92636935),new n.Ar(42,72.925169832),new n.Ar(43,73.926940999),new n.Ar(44,74.926500645),new n.Ar(45,75.928928262),new n.Ar(46,76.929281189),new n.Ar(47,77.93165595),new n.Ar(48,78.932916371),new n.Ar(49,79.936588154),new n.Ar(50,80.937752955),new n.Ar(51,81.94316),new n.Ar(52,82.94687),new n.Ar(53,83.95234)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(26,57.99101),new n.Ar(27,58.98175),new n.Ar(28,59.97019),new n.Ar(29,60.96379),new n.Ar(30,61.95465),new n.Ar(31,62.94964),new n.Ar(32,63.941572638),new n.Ar(33,64.939440762),new n.Ar(34,65.933846798),new n.Ar(35,66.932738415),new n.Ar(36,67.928097266),new n.Ar(37,68.927972002),new n.Ar(38,69.924250365),new n.Ar(39,70.924953991),new n.Ar(40,71.922076184),new n.Ar(41,72.923459361),new n.Ar(42,73.921178213),new n.Ar(43,74.922859494),new n.Ar(44,75.921402716),new n.Ar(45,76.923548462),new n.Ar(46,77.922852886),new n.Ar(47,78.92540156),new n.Ar(48,79.925444764),new n.Ar(49,80.928821065),new n.Ar(50,81.929550326),new n.Ar(51,82.93451),new n.Ar(52,83.93731),new n.Ar(53,84.94269),new n.Ar(54,85.94627)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(27,59.99313),new n.Ar(28,60.98062),new n.Ar(29,61.9732),new n.Ar(30,62.96369),new n.Ar(31,63.957572),new n.Ar(32,64.949484),new n.Ar(33,65.944099147),new n.Ar(34,66.939190417),new n.Ar(35,67.936792976),new n.Ar(36,68.932280154),new n.Ar(37,69.930927811),new n.Ar(38,70.927114724),new n.Ar(39,71.926752647),new n.Ar(40,72.923825288),new n.Ar(41,73.923929076),new n.Ar(42,74.921596417),new n.Ar(43,75.922393933),new n.Ar(44,76.920647703),new n.Ar(45,77.921828577),new n.Ar(46,78.920948498),new n.Ar(47,79.922578162),new n.Ar(48,80.922132884),new n.Ar(49,81.924504668),new n.Ar(50,82.924980625),new n.Ar(51,83.92906),new n.Ar(52,84.93181),new n.Ar(53,85.93623),new n.Ar(54,86.93958),new n.Ar(55,87.94456),new n.Ar(56,88.94923)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(31,64.96466),new n.Ar(32,65.95521),new n.Ar(33,66.95009),new n.Ar(34,67.94187),new n.Ar(35,68.939562155),new n.Ar(36,69.933504),new n.Ar(37,70.931868378),new n.Ar(38,71.927112313),new n.Ar(39,72.9267668),new n.Ar(40,73.922476561),new n.Ar(41,74.922523571),new n.Ar(42,75.919214107),new n.Ar(43,76.91991461),new n.Ar(44,77.917309522),new n.Ar(45,78.918499802),new n.Ar(46,79.916521828),new n.Ar(47,80.917992931),new n.Ar(48,81.9167),new n.Ar(49,82.919119072),new n.Ar(50,83.918464523),new n.Ar(51,84.922244678),new n.Ar(52,85.924271165),new n.Ar(53,86.928520749),new n.Ar(54,87.931423982),new n.Ar(55,88.93602),new n.Ar(56,89.93942),new n.Ar(57,90.94537),new n.Ar(58,91.94933)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(32,66.96479),new n.Ar(33,67.958248),new n.Ar(34,68.950178),new n.Ar(35,69.944208),new n.Ar(36,70.939246),new n.Ar(37,71.936496876),new n.Ar(38,72.931794889),new n.Ar(39,73.929891152),new n.Ar(40,74.92577641),new n.Ar(41,75.924541974),new n.Ar(42,76.921380123),new n.Ar(43,77.92114613),new n.Ar(44,78.918337647),new n.Ar(45,79.918529952),new n.Ar(46,80.91629106),new n.Ar(47,81.916804666),new n.Ar(48,82.915180219),new n.Ar(49,83.916503685),new n.Ar(50,84.915608027),new n.Ar(51,85.918797162),new n.Ar(52,86.920710713),new n.Ar(53,87.924065908),new n.Ar(54,88.92638726),new n.Ar(55,89.930634988),new n.Ar(56,90.9339653),new n.Ar(57,91.939255258),new n.Ar(58,92.9431),new n.Ar(59,93.94868)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(33,68.96532),new n.Ar(34,69.95601),new n.Ar(35,70.95051),new n.Ar(36,71.94190754),new n.Ar(37,72.938931115),new n.Ar(38,73.933258225),new n.Ar(39,74.931033794),new n.Ar(40,75.925948304),new n.Ar(41,76.92466788),new n.Ar(42,77.920386271),new n.Ar(43,78.920082992),new n.Ar(44,79.91637804),new n.Ar(45,80.916592419),new n.Ar(46,81.913484601),new n.Ar(47,82.914135952),new n.Ar(48,83.911506627),new n.Ar(49,84.912526954),new n.Ar(50,85.910610313),new n.Ar(51,86.913354251),new n.Ar(52,87.914446951),new n.Ar(53,88.917632505),new n.Ar(54,89.919523803),new n.Ar(55,90.923442418),new n.Ar(56,91.926152752),new n.Ar(57,92.931265246),new n.Ar(58,93.934362),new n.Ar(59,94.93984),new n.Ar(60,95.94307),new n.Ar(61,96.94856)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(34,70.96532),new n.Ar(35,71.95908),new n.Ar(36,72.950366),new n.Ar(37,73.944470376),new n.Ar(38,74.938569199),new n.Ar(39,75.935071448),new n.Ar(40,76.930406599),new n.Ar(41,77.928141485),new n.Ar(42,78.923996719),new n.Ar(43,79.922519322),new n.Ar(44,80.918994165),new n.Ar(45,81.918207691),new n.Ar(46,82.915111951),new n.Ar(47,83.914384676),new n.Ar(48,84.911789341),new n.Ar(49,85.91116708),new n.Ar(50,86.909183465),new n.Ar(51,87.911318556),new n.Ar(52,88.912279939),new n.Ar(53,89.914808941),new n.Ar(54,90.91653416),new n.Ar(55,91.919725442),new n.Ar(56,92.922032765),new n.Ar(57,93.926407326),new n.Ar(58,94.92931926),new n.Ar(59,95.934283962),new n.Ar(60,96.937342863),new n.Ar(61,97.941703557),new n.Ar(62,98.945420616),new n.Ar(63,99.94987),new n.Ar(64,100.953195994),new n.Ar(65,101.95921)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(35,72.96597),new n.Ar(36,73.95631),new n.Ar(37,74.94992),new n.Ar(38,75.94161),new n.Ar(39,76.937761511),new n.Ar(40,77.932179362),new n.Ar(41,78.929707076),new n.Ar(42,79.924524588),new n.Ar(43,80.923213095),new n.Ar(44,81.918401258),new n.Ar(45,82.917555029),new n.Ar(46,83.913424778),new n.Ar(47,84.912932689),new n.Ar(48,85.909262351),new n.Ar(49,86.908879316),new n.Ar(50,87.905614339),new n.Ar(51,88.907452906),new n.Ar(52,89.907737596),new n.Ar(53,90.910209845),new n.Ar(54,91.911029895),new n.Ar(55,92.91402241),new n.Ar(56,93.915359856),new n.Ar(57,94.919358213),new n.Ar(58,95.921680473),new n.Ar(59,96.926148757),new n.Ar(60,97.928471177),new n.Ar(61,98.933315038),new n.Ar(62,99.935351729),new n.Ar(63,100.940517434),new n.Ar(64,101.943018795),new n.Ar(65,102.94895),new n.Ar(66,103.95233)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(38,76.94962),new n.Ar(39,77.9435),new n.Ar(40,78.937350712),new n.Ar(41,79.931982402),new n.Ar(42,80.929128719),new n.Ar(43,81.926792071),new n.Ar(44,82.922352572),new n.Ar(45,83.920387768),new n.Ar(46,84.916427076),new n.Ar(47,85.914887724),new n.Ar(48,86.910877833),new n.Ar(49,87.909503361),new n.Ar(50,88.905847902),new n.Ar(51,89.907151443),new n.Ar(52,90.907303415),new n.Ar(53,91.908946832),new n.Ar(54,92.909581582),new n.Ar(55,93.911594008),new n.Ar(56,94.912823709),new n.Ar(57,95.915897787),new n.Ar(58,96.918131017),new n.Ar(59,97.922219525),new n.Ar(60,98.924634736),new n.Ar(61,99.927756402),new n.Ar(62,100.930313395),new n.Ar(63,101.933555501),new n.Ar(64,102.93694),new n.Ar(65,103.94145),new n.Ar(66,104.94509),new n.Ar(67,105.95022)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(39,78.94916),new n.Ar(40,79.94055),new n.Ar(41,80.936815296),new n.Ar(42,81.931086249),new n.Ar(43,82.92865213),new n.Ar(44,83.92325),new n.Ar(45,84.92146522),new n.Ar(46,85.916472851),new n.Ar(47,86.914816578),new n.Ar(48,87.910226179),new n.Ar(49,88.908888916),new n.Ar(50,89.904703679),new n.Ar(51,90.905644968),new n.Ar(52,91.905040106),new n.Ar(53,92.906475627),new n.Ar(54,93.906315765),new n.Ar(55,94.908042739),new n.Ar(56,95.908275675),new n.Ar(57,96.910950716),new n.Ar(58,97.912746366),new n.Ar(59,98.916511084),new n.Ar(60,99.917761704),new n.Ar(61,100.921139958),new n.Ar(62,101.922981089),new n.Ar(63,102.926597062),new n.Ar(64,103.92878),new n.Ar(65,104.93305),new n.Ar(66,105.93591),new n.Ar(67,106.94086),new n.Ar(68,107.94428)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(40,80.94905),new n.Ar(41,81.94313),new n.Ar(42,82.936703713),new n.Ar(43,83.93357),new n.Ar(44,84.927906486),new n.Ar(45,85.925037588),new n.Ar(46,86.920361435),new n.Ar(47,87.91833144),new n.Ar(48,88.913495503),new n.Ar(49,89.911264109),new n.Ar(50,90.906990538),new n.Ar(51,91.907193214),new n.Ar(52,92.906377543),new n.Ar(53,93.907283457),new n.Ar(54,94.906835178),new n.Ar(55,95.908100076),new n.Ar(56,96.908097144),new n.Ar(57,97.91033069),new n.Ar(58,98.911617864),new n.Ar(59,99.914181434),new n.Ar(60,100.915251567),new n.Ar(61,101.918037417),new n.Ar(62,102.919141297),new n.Ar(63,103.922459464),new n.Ar(64,104.923934023),new n.Ar(65,105.92819),new n.Ar(66,106.93031),new n.Ar(67,107.93501),new n.Ar(68,108.93763),new n.Ar(69,109.94268)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(41,82.94874),new n.Ar(42,83.94009),new n.Ar(43,84.93659),new n.Ar(44,85.930695167),new n.Ar(45,86.92732683),new n.Ar(46,87.921952728),new n.Ar(47,88.919480562),new n.Ar(48,89.913936161),new n.Ar(49,90.911750754),new n.Ar(50,91.90681048),new n.Ar(51,92.906812213),new n.Ar(52,93.905087578),new n.Ar(53,94.905841487),new n.Ar(54,95.904678904),new n.Ar(55,96.906021033),new n.Ar(56,97.905407846),new n.Ar(57,98.907711598),new n.Ar(58,99.907477149),new n.Ar(59,100.910346543),new n.Ar(60,101.910297162),new n.Ar(61,102.913204596),new n.Ar(62,103.913758387),new n.Ar(63,104.916972087),new n.Ar(64,105.918134284),new n.Ar(65,106.921694724),new n.Ar(66,107.923973837),new n.Ar(67,108.92781),new n.Ar(68,109.92973),new n.Ar(69,110.93451),new n.Ar(70,111.93684),new n.Ar(71,112.94203)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(42,84.94894),new n.Ar(43,85.94288),new n.Ar(44,86.93653),new n.Ar(45,87.93283),new n.Ar(46,88.92754288),new n.Ar(47,89.92355583),new n.Ar(48,90.9184282),new n.Ar(49,91.915259655),new n.Ar(50,92.910248473),new n.Ar(51,93.909656309),new n.Ar(52,94.907656454),new n.Ar(53,95.907870803),new n.Ar(54,96.906364843),new n.Ar(55,97.907215692),new n.Ar(56,98.906254554),new n.Ar(57,99.907657594),new n.Ar(58,100.90731438),new n.Ar(59,101.909212938),new n.Ar(60,102.909178805),new n.Ar(61,103.911444898),new n.Ar(62,104.911658043),new n.Ar(63,105.914355408),new n.Ar(64,106.915081691),new n.Ar(65,107.918479973),new n.Ar(66,108.919980998),new n.Ar(67,109.92339),new n.Ar(68,110.92505),new n.Ar(69,111.92924),new n.Ar(70,112.93133),new n.Ar(71,113.93588),new n.Ar(72,114.93828)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(43,86.94918),new n.Ar(44,87.94042),new n.Ar(45,88.93611),new n.Ar(46,89.92978),new n.Ar(47,90.926377434),new n.Ar(48,91.92012),new n.Ar(49,92.917051523),new n.Ar(50,93.911359569),new n.Ar(51,94.910412729),new n.Ar(52,95.907597681),new n.Ar(53,96.907554546),new n.Ar(54,97.905287111),new n.Ar(55,98.905939307),new n.Ar(56,99.904219664),new n.Ar(57,100.905582219),new n.Ar(58,101.904349503),new n.Ar(59,102.906323677),new n.Ar(60,103.905430145),new n.Ar(61,104.907750341),new n.Ar(62,105.907326913),new n.Ar(63,106.909907207),new n.Ar(64,107.910192211),new n.Ar(65,108.913201565),new n.Ar(66,109.913966185),new n.Ar(67,110.91756),new n.Ar(68,111.918821673),new n.Ar(69,112.92254),new n.Ar(70,113.923891981),new n.Ar(71,114.92831),new n.Ar(72,115.93016),new n.Ar(73,116.93479),new n.Ar(74,117.93703)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(44,88.94938),new n.Ar(45,89.94287),new n.Ar(46,90.93655),new n.Ar(47,91.93198),new n.Ar(48,92.92574),new n.Ar(49,93.921698),new n.Ar(50,94.915898541),new n.Ar(51,95.914518212),new n.Ar(52,96.911336643),new n.Ar(53,97.910716431),new n.Ar(54,98.908132101),new n.Ar(55,99.90811663),new n.Ar(56,100.906163526),new n.Ar(57,101.906842845),new n.Ar(58,102.905504182),new n.Ar(59,103.906655315),new n.Ar(60,104.905692444),new n.Ar(61,105.907284615),new n.Ar(62,106.90675054),new n.Ar(63,107.908730768),new n.Ar(64,108.908735621),new n.Ar(65,109.910949525),new n.Ar(66,110.91166),new n.Ar(67,111.913969253),new n.Ar(68,112.91542),new n.Ar(69,113.91734336),new n.Ar(70,114.920124676),new n.Ar(71,115.922746643),new n.Ar(72,116.92535),new n.Ar(73,117.92943),new n.Ar(74,118.93136),new n.Ar(75,119.93578),new n.Ar(76,120.93808)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(45,90.94948),new n.Ar(46,91.94042),new n.Ar(47,92.93591),new n.Ar(48,93.92877),new n.Ar(49,94.92469),new n.Ar(50,95.91822194),new n.Ar(51,96.916478921),new n.Ar(52,97.912720751),new n.Ar(53,98.911767757),new n.Ar(54,99.908504596),new n.Ar(55,100.908289144),new n.Ar(56,101.905607716),new n.Ar(57,102.906087204),new n.Ar(58,103.904034912),new n.Ar(59,104.905084046),new n.Ar(60,105.903483087),new n.Ar(61,106.905128453),new n.Ar(62,107.903894451),new n.Ar(63,108.905953535),new n.Ar(64,109.905152385),new n.Ar(65,110.907643952),new n.Ar(66,111.907313277),new n.Ar(67,112.910151346),new n.Ar(68,113.910365322),new n.Ar(69,114.91368341),new n.Ar(70,115.914158288),new n.Ar(71,116.91784),new n.Ar(72,117.918983915),new n.Ar(73,118.92268),new n.Ar(74,119.92403),new n.Ar(75,120.92818),new n.Ar(76,121.9298),new n.Ar(77,122.93426)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(47,93.94278),new n.Ar(48,94.93548),new n.Ar(49,95.93068),new n.Ar(50,96.924),new n.Ar(51,97.921759995),new n.Ar(52,98.917597103),new n.Ar(53,99.916069387),new n.Ar(54,100.912802135),new n.Ar(55,101.911999996),new n.Ar(56,102.908972453),new n.Ar(57,103.908628228),new n.Ar(58,104.906528234),new n.Ar(59,105.906666431),new n.Ar(60,106.90509302),new n.Ar(61,107.905953705),new n.Ar(62,108.904755514),new n.Ar(63,109.90611046),new n.Ar(64,110.905294679),new n.Ar(65,111.907004132),new n.Ar(66,112.906565708),new n.Ar(67,113.908807907),new n.Ar(68,114.908762282),new n.Ar(69,115.911359558),new n.Ar(70,116.911684187),new n.Ar(71,117.914582383),new n.Ar(72,118.915666045),new n.Ar(73,119.918788609),new n.Ar(74,120.919851074),new n.Ar(75,121.92332),new n.Ar(76,122.9249),new n.Ar(77,123.92853),new n.Ar(78,124.93054),new n.Ar(79,125.9345),new n.Ar(80,126.93688)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(48,95.93977),new n.Ar(49,96.93494),new n.Ar(50,97.927579),new n.Ar(51,98.92501),new n.Ar(52,99.920230232),new n.Ar(53,100.918681442),new n.Ar(54,101.914777255),new n.Ar(55,102.913418952),new n.Ar(56,103.909848091),new n.Ar(57,104.909467818),new n.Ar(58,105.906458007),new n.Ar(59,106.906614232),new n.Ar(60,107.904183403),new n.Ar(61,108.904985569),new n.Ar(62,109.903005578),new n.Ar(63,110.904181628),new n.Ar(64,111.902757226),new n.Ar(65,112.904400947),new n.Ar(66,113.903358121),new n.Ar(67,114.905430553),new n.Ar(68,115.904755434),new n.Ar(69,116.907218242),new n.Ar(70,117.906914144),new n.Ar(71,118.909922582),new n.Ar(72,119.909851352),new n.Ar(73,120.91298039),new n.Ar(74,121.9135),new n.Ar(75,122.917003675),new n.Ar(76,123.917648302),new n.Ar(77,124.92124717),new n.Ar(78,125.922353996),new n.Ar(79,126.926434822),new n.Ar(80,127.927760617),new n.Ar(81,128.93226),new n.Ar(82,129.93398)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(49,97.94224),new n.Ar(50,98.93461),new n.Ar(51,99.931149033),new n.Ar(52,100.92656),new n.Ar(53,101.924707541),new n.Ar(54,102.919913896),new n.Ar(55,103.918338416),new n.Ar(56,104.914673434),new n.Ar(57,105.913461134),new n.Ar(58,106.910292195),new n.Ar(59,107.909719683),new n.Ar(60,108.907154078),new n.Ar(61,109.907168783),new n.Ar(62,110.905110677),new n.Ar(63,111.905533338),new n.Ar(64,112.904061223),new n.Ar(65,113.904916758),new n.Ar(66,114.903878328),new n.Ar(67,115.905259995),new n.Ar(68,116.904515731),new n.Ar(69,117.906354623),new n.Ar(70,118.905846334),new n.Ar(71,119.907961505),new n.Ar(72,120.907848847),new n.Ar(73,121.910277103),new n.Ar(74,122.910438951),new n.Ar(75,123.913175916),new n.Ar(76,124.913601387),new n.Ar(77,125.916464532),new n.Ar(78,126.917344048),new n.Ar(79,127.920170658),new n.Ar(80,128.921657958),new n.Ar(81,129.924854941),new n.Ar(82,130.926767408),new n.Ar(83,131.932919005),new n.Ar(84,132.93834),new n.Ar(85,133.94466)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(50,99.938954),new n.Ar(51,100.93606),new n.Ar(52,101.93049),new n.Ar(53,102.92813),new n.Ar(54,103.923185469),new n.Ar(55,104.921390409),new n.Ar(56,105.916880472),new n.Ar(57,106.915666702),new n.Ar(58,107.911965339),new n.Ar(59,108.911286879),new n.Ar(60,109.907852688),new n.Ar(61,110.907735404),new n.Ar(62,111.90482081),new n.Ar(63,112.905173373),new n.Ar(64,113.902781816),new n.Ar(65,114.903345973),new n.Ar(66,115.901744149),new n.Ar(67,116.902953765),new n.Ar(68,117.901606328),new n.Ar(69,118.90330888),new n.Ar(70,119.902196571),new n.Ar(71,120.904236867),new n.Ar(72,121.903440138),new n.Ar(73,122.905721901),new n.Ar(74,123.90527463),new n.Ar(75,124.907784924),new n.Ar(76,125.907653953),new n.Ar(77,126.91035098),new n.Ar(78,127.910534953),new n.Ar(79,128.913439976),new n.Ar(80,129.913852185),new n.Ar(81,130.916919144),new n.Ar(82,131.917744455),new n.Ar(83,132.923814085),new n.Ar(84,133.928463576),new n.Ar(85,134.93473),new n.Ar(86,135.93934),new n.Ar(87,136.94579)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(52,102.94012),new n.Ar(53,103.936287),new n.Ar(54,104.931528593),new n.Ar(55,105.928183134),new n.Ar(56,106.92415),new n.Ar(57,107.92216),new n.Ar(58,108.918136092),new n.Ar(59,109.917533911),new n.Ar(60,110.912534147),new n.Ar(61,111.91239464),new n.Ar(62,112.909377941),new n.Ar(63,113.909095876),new n.Ar(64,114.906598812),new n.Ar(65,115.906797235),new n.Ar(66,116.90483959),new n.Ar(67,117.905531885),new n.Ar(68,118.90394646),new n.Ar(69,119.905074315),new n.Ar(70,120.903818044),new n.Ar(71,121.905175415),new n.Ar(72,122.904215696),new n.Ar(73,123.905937525),new n.Ar(74,124.905247804),new n.Ar(75,125.907248153),new n.Ar(76,126.906914564),new n.Ar(77,127.90916733),new n.Ar(78,128.909150092),new n.Ar(79,129.911546459),new n.Ar(80,130.911946487),new n.Ar(81,131.914413247),new n.Ar(82,132.915236466),new n.Ar(83,133.920551554),new n.Ar(84,134.925167962),new n.Ar(85,135.93066),new n.Ar(86,136.93531),new n.Ar(87,137.94096),new n.Ar(88,138.94571)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(54,105.937702),new n.Ar(55,106.935036),new n.Ar(56,107.929486838),new n.Ar(57,108.927456483),new n.Ar(58,109.922407164),new n.Ar(59,110.921120589),new n.Ar(60,111.917061617),new n.Ar(61,112.915452551),new n.Ar(62,113.912498025),new n.Ar(63,114.911578627),new n.Ar(64,115.908420253),new n.Ar(65,116.90863418),new n.Ar(66,117.905825187),new n.Ar(67,118.90640811),new n.Ar(68,119.904019891),new n.Ar(69,120.904929815),new n.Ar(70,121.903047064),new n.Ar(71,122.904272951),new n.Ar(72,123.902819466),new n.Ar(73,124.904424718),new n.Ar(74,125.903305543),new n.Ar(75,126.90521729),new n.Ar(76,127.904461383),new n.Ar(77,128.906595593),new n.Ar(78,129.906222753),new n.Ar(79,130.90852188),new n.Ar(80,131.908523782),new n.Ar(81,132.910939068),new n.Ar(82,133.911540546),new n.Ar(83,134.916450782),new n.Ar(84,135.920103155),new n.Ar(85,136.925324769),new n.Ar(86,137.92922),new n.Ar(87,138.93473),new n.Ar(88,139.9387),new n.Ar(89,140.94439),new n.Ar(90,141.9485)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(55,107.943291),new n.Ar(56,108.938191658),new n.Ar(57,109.934634181),new n.Ar(58,110.930276),new n.Ar(59,111.92797),new n.Ar(60,112.923644245),new n.Ar(61,113.92185),new n.Ar(62,114.918272),new n.Ar(63,115.916735014),new n.Ar(64,116.913647692),new n.Ar(65,117.91337523),new n.Ar(66,118.910180837),new n.Ar(67,119.910047843),new n.Ar(68,120.907366063),new n.Ar(69,121.907592451),new n.Ar(70,122.905597944),new n.Ar(71,123.906211423),new n.Ar(72,124.90462415),new n.Ar(73,125.905619387),new n.Ar(74,126.90446842),new n.Ar(75,127.905805254),new n.Ar(76,128.904987487),new n.Ar(77,129.906674018),new n.Ar(78,130.906124168),new n.Ar(79,131.907994525),new n.Ar(80,132.907806465),new n.Ar(81,133.909876552),new n.Ar(82,134.91005031),new n.Ar(83,135.914655105),new n.Ar(84,136.917872653),new n.Ar(85,137.922383666),new n.Ar(86,138.926093402),new n.Ar(87,139.93121),new n.Ar(88,140.93483),new n.Ar(89,141.94018),new n.Ar(90,142.94407),new n.Ar(91,143.94961)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(56,109.944476),new n.Ar(57,110.941632),new n.Ar(58,111.93566535),new n.Ar(59,112.933382836),new n.Ar(60,113.928145),new n.Ar(61,114.926979032),new n.Ar(62,115.921394197),new n.Ar(63,116.920564355),new n.Ar(64,117.91657092),new n.Ar(65,118.915554295),new n.Ar(66,119.91215199),new n.Ar(67,120.911386497),new n.Ar(68,121.908548396),new n.Ar(69,122.908470748),new n.Ar(70,123.905895774),new n.Ar(71,124.906398236),new n.Ar(72,125.904268868),new n.Ar(73,126.905179581),new n.Ar(74,127.903530436),new n.Ar(75,128.904779458),new n.Ar(76,129.903507903),new n.Ar(77,130.90508192),new n.Ar(78,131.904154457),new n.Ar(79,132.90590566),new n.Ar(80,133.905394504),new n.Ar(81,134.907207499),new n.Ar(82,135.907219526),new n.Ar(83,136.911562939),new n.Ar(84,137.913988549),new n.Ar(85,138.918786859),new n.Ar(86,139.921635665),new n.Ar(87,140.926646282),new n.Ar(88,141.929702981),new n.Ar(89,142.93489),new n.Ar(90,143.93823),new n.Ar(91,144.94367),new n.Ar(92,145.9473),new n.Ar(93,146.95301)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(57,111.950331),new n.Ar(58,112.944535512),new n.Ar(59,113.940841319),new n.Ar(60,114.935939),new n.Ar(61,115.932914152),new n.Ar(62,116.928639484),new n.Ar(63,117.926554883),new n.Ar(64,118.922370879),new n.Ar(65,119.920678219),new n.Ar(66,120.917183637),new n.Ar(67,121.916121946),new n.Ar(68,122.912990168),new n.Ar(69,123.912245731),new n.Ar(70,124.909724871),new n.Ar(71,125.909447953),new n.Ar(72,126.9074176),new n.Ar(73,127.907747919),new n.Ar(74,128.906063369),new n.Ar(75,129.906706163),new n.Ar(76,130.905460232),new n.Ar(77,131.906429799),new n.Ar(78,132.90544687),new n.Ar(79,133.906713419),new n.Ar(80,134.905971903),new n.Ar(81,135.907305741),new n.Ar(82,136.907083505),new n.Ar(83,137.911010537),new n.Ar(84,138.913357921),new n.Ar(85,139.917277075),new n.Ar(86,140.920043984),new n.Ar(87,141.924292317),new n.Ar(88,142.927330292),new n.Ar(89,143.932027373),new n.Ar(90,144.935388226),new n.Ar(91,145.940162028),new n.Ar(92,146.943864435),new n.Ar(93,147.948899539),new n.Ar(94,148.95272),new n.Ar(95,149.95797),new n.Ar(96,150.962)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(58,113.950941),new n.Ar(59,114.94771),new n.Ar(60,115.94168),new n.Ar(61,116.937700229),new n.Ar(62,117.93344),new n.Ar(63,118.931051927),new n.Ar(64,119.926045941),new n.Ar(65,120.924485908),new n.Ar(66,121.92026),new n.Ar(67,122.91885),new n.Ar(68,123.915088437),new n.Ar(69,124.914620234),new n.Ar(70,125.911244146),new n.Ar(71,126.911121328),new n.Ar(72,127.90830887),new n.Ar(73,128.908673749),new n.Ar(74,129.906310478),new n.Ar(75,130.906930798),new n.Ar(76,131.905056152),new n.Ar(77,132.906002368),new n.Ar(78,133.904503347),new n.Ar(79,134.905682749),new n.Ar(80,135.904570109),new n.Ar(81,136.905821414),new n.Ar(82,137.905241273),new n.Ar(83,138.908835384),new n.Ar(84,139.910599485),new n.Ar(85,140.914406439),new n.Ar(86,141.916448175),new n.Ar(87,142.920617184),new n.Ar(88,143.922940468),new n.Ar(89,144.926923807),new n.Ar(90,145.930106645),new n.Ar(91,146.933992519),new n.Ar(92,147.937682377),new n.Ar(93,148.94246),new n.Ar(94,149.94562),new n.Ar(95,150.9507),new n.Ar(96,151.95416),new n.Ar(97,152.95961)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(60,116.95001),new n.Ar(61,117.94657),new n.Ar(62,118.94099),new n.Ar(63,119.93807),new n.Ar(64,120.93301),new n.Ar(65,121.93071),new n.Ar(66,122.92624),new n.Ar(67,123.92453),new n.Ar(68,124.92067),new n.Ar(69,125.91937),new n.Ar(70,126.91616),new n.Ar(71,127.91544794),new n.Ar(72,128.912667334),new n.Ar(73,129.91232),new n.Ar(74,130.910108489),new n.Ar(75,131.910110399),new n.Ar(76,132.908396372),new n.Ar(77,133.908489607),new n.Ar(78,134.906971003),new n.Ar(79,135.907651181),new n.Ar(80,136.906465656),new n.Ar(81,137.907106826),new n.Ar(82,138.90634816),new n.Ar(83,139.909472552),new n.Ar(84,140.910957016),new n.Ar(85,141.914074489),new n.Ar(86,142.916058646),new n.Ar(87,143.919591666),new n.Ar(88,144.92163837),new n.Ar(89,145.925700146),new n.Ar(90,146.927819639),new n.Ar(91,147.932191197),new n.Ar(92,148.93437),new n.Ar(93,149.93857),new n.Ar(94,150.94156),new n.Ar(95,151.94611),new n.Ar(96,152.94945),new n.Ar(97,153.9544),new n.Ar(98,154.95813)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(61,118.95276),new n.Ar(62,119.94664),new n.Ar(63,120.94367),new n.Ar(64,121.93801),new n.Ar(65,122.93551),new n.Ar(66,123.93052),new n.Ar(67,124.92854),new n.Ar(68,125.9241),new n.Ar(69,126.92275),new n.Ar(70,127.91887),new n.Ar(71,128.918679183),new n.Ar(72,129.914339361),new n.Ar(73,130.914424137),new n.Ar(74,131.91149),new n.Ar(75,132.91155),new n.Ar(76,133.909026379),new n.Ar(77,134.909145555),new n.Ar(78,135.907143574),new n.Ar(79,136.907777634),new n.Ar(80,137.905985574),new n.Ar(81,138.906646605),new n.Ar(82,139.905434035),new n.Ar(83,140.908271103),new n.Ar(84,141.909239733),new n.Ar(85,142.912381158),new n.Ar(86,143.913642686),new n.Ar(87,144.917227871),new n.Ar(88,145.918689722),new n.Ar(89,146.922510962),new n.Ar(90,147.924394738),new n.Ar(91,148.928289207),new n.Ar(92,149.930226399),new n.Ar(93,150.93404),new n.Ar(94,151.93638),new n.Ar(95,152.94058),new n.Ar(96,153.94332),new n.Ar(97,154.94804),new n.Ar(98,155.95126),new n.Ar(99,156.95634)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(62,120.955364),new n.Ar(63,121.95165),new n.Ar(64,122.94596),new n.Ar(65,123.94296),new n.Ar(66,124.93783),new n.Ar(67,125.93531),new n.Ar(68,126.93083),new n.Ar(69,127.9288),new n.Ar(70,128.92486),new n.Ar(71,129.92338),new n.Ar(72,130.920060245),new n.Ar(73,131.91912),new n.Ar(74,132.9162),new n.Ar(75,133.915672),new n.Ar(76,134.91313914),new n.Ar(77,135.912646935),new n.Ar(78,136.910678351),new n.Ar(79,137.910748891),new n.Ar(80,138.908932181),new n.Ar(81,139.909071204),new n.Ar(82,140.907647726),new n.Ar(83,141.910039865),new n.Ar(84,142.910812233),new n.Ar(85,143.913300595),new n.Ar(86,144.914506897),new n.Ar(87,145.917588016),new n.Ar(88,146.918979001),new n.Ar(89,147.922183237),new n.Ar(90,148.923791056),new n.Ar(91,149.926995031),new n.Ar(92,150.928227869),new n.Ar(93,151.9316),new n.Ar(94,152.93365),new n.Ar(95,153.93739),new n.Ar(96,154.93999),new n.Ar(97,155.94412),new n.Ar(98,156.94717),new n.Ar(99,157.95178),new n.Ar(100,158.95523)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(66,125.94307),new n.Ar(67,126.9405),new n.Ar(68,127.93539),new n.Ar(69,128.932385),new n.Ar(70,129.92878),new n.Ar(71,130.927102697),new n.Ar(72,131.92312),new n.Ar(73,132.92221),new n.Ar(74,133.918645),new n.Ar(75,134.91824),new n.Ar(76,135.915020542),new n.Ar(77,136.91463973),new n.Ar(78,137.91291745),new n.Ar(79,138.91192415),new n.Ar(80,139.909309824),new n.Ar(81,140.9096048),new n.Ar(82,141.907718643),new n.Ar(83,142.909809626),new n.Ar(84,143.910082629),new n.Ar(85,144.912568847),new n.Ar(86,145.913112139),new n.Ar(87,146.916095794),new n.Ar(88,147.916888516),new n.Ar(89,148.92014419),new n.Ar(90,149.920886563),new n.Ar(91,150.923824739),new n.Ar(92,151.924682428),new n.Ar(93,152.927694534),new n.Ar(94,153.929483295),new n.Ar(95,154.932629551),new n.Ar(96,155.9352),new n.Ar(97,156.93927),new n.Ar(98,157.94187),new n.Ar(99,158.94639),new n.Ar(100,159.94939),new n.Ar(101,160.95433)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(67,127.94826),new n.Ar(68,128.94316),new n.Ar(69,129.94045),new n.Ar(70,130.9358),new n.Ar(71,131.93375),new n.Ar(72,132.92972),new n.Ar(73,133.92849),new n.Ar(74,134.924617),new n.Ar(75,135.923447865),new n.Ar(76,136.920713),new n.Ar(77,137.920432261),new n.Ar(78,138.916759814),new n.Ar(79,139.915801649),new n.Ar(80,140.913606636),new n.Ar(81,141.912950738),new n.Ar(82,142.910927571),new n.Ar(83,143.912585768),new n.Ar(84,144.912743879),new n.Ar(85,145.914692165),new n.Ar(86,146.915133898),new n.Ar(87,147.917467786),new n.Ar(88,148.918329195),new n.Ar(89,149.920979477),new n.Ar(90,150.921202693),new n.Ar(91,151.923490557),new n.Ar(92,152.924113189),new n.Ar(93,153.926547019),new n.Ar(94,154.928097047),new n.Ar(95,155.931060357),new n.Ar(96,156.9332),new n.Ar(97,157.93669),new n.Ar(98,158.93913),new n.Ar(99,159.94299),new n.Ar(100,160.94586),new n.Ar(101,161.95029),new n.Ar(102,162.95352)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(68,129.94863),new n.Ar(69,130.94589),new n.Ar(70,131.94082),new n.Ar(71,132.93873),new n.Ar(72,133.93402),new n.Ar(73,134.93235),new n.Ar(74,135.9283),new n.Ar(75,136.927046709),new n.Ar(76,137.92354),new n.Ar(77,138.922302),new n.Ar(78,139.918991),new n.Ar(79,140.918468512),new n.Ar(80,141.915193274),new n.Ar(81,142.914623555),new n.Ar(82,143.91199473),new n.Ar(83,144.913405611),new n.Ar(84,145.91303676),new n.Ar(85,146.914893275),new n.Ar(86,147.914817914),new n.Ar(87,148.917179521),new n.Ar(88,149.917271454),new n.Ar(89,150.919928351),new n.Ar(90,151.919728244),new n.Ar(91,152.922093907),new n.Ar(92,153.922205303),new n.Ar(93,154.92463594),new n.Ar(94,155.925526236),new n.Ar(95,156.928354506),new n.Ar(96,157.929987938),new n.Ar(97,158.9332),new n.Ar(98,159.93514),new n.Ar(99,160.93883),new n.Ar(100,161.94122),new n.Ar(101,162.94536),new n.Ar(102,163.94828),new n.Ar(103,164.95298)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(69,131.95416),new n.Ar(70,132.9489),new n.Ar(71,133.94632),new n.Ar(72,134.94172),new n.Ar(73,135.9395),new n.Ar(74,136.93521),new n.Ar(75,137.93345),new n.Ar(76,138.92882915),new n.Ar(77,139.928083921),new n.Ar(78,140.924885867),new n.Ar(79,141.923400033),new n.Ar(80,142.920286634),new n.Ar(81,143.918774116),new n.Ar(82,144.916261285),new n.Ar(83,145.917199714),new n.Ar(84,146.916741206),new n.Ar(85,147.918153775),new n.Ar(86,148.917925922),new n.Ar(87,149.919698294),new n.Ar(88,150.919846022),new n.Ar(89,151.921740399),new n.Ar(90,152.921226219),new n.Ar(91,153.922975386),new n.Ar(92,154.922889429),new n.Ar(93,155.924750855),new n.Ar(94,156.925419435),new n.Ar(95,157.927841923),new n.Ar(96,158.9290845),new n.Ar(97,159.931460406),new n.Ar(98,160.93368),new n.Ar(99,161.93704),new n.Ar(100,162.93921),new n.Ar(101,163.94299),new n.Ar(102,164.94572),new n.Ar(103,165.94997),new n.Ar(104,166.95305)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(72,135.94707),new n.Ar(73,136.94465),new n.Ar(74,137.93997),new n.Ar(75,138.93808),new n.Ar(76,139.933236934),new n.Ar(77,140.93221),new n.Ar(78,141.927908919),new n.Ar(79,142.926738636),new n.Ar(80,143.923390357),new n.Ar(81,144.921687498),new n.Ar(82,145.918305344),new n.Ar(83,146.919089446),new n.Ar(84,147.918109771),new n.Ar(85,148.919336427),new n.Ar(86,149.918655455),new n.Ar(87,150.920344273),new n.Ar(88,151.919787882),new n.Ar(89,152.921746283),new n.Ar(90,153.920862271),new n.Ar(91,154.922618801),new n.Ar(92,155.922119552),new n.Ar(93,156.923956686),new n.Ar(94,157.924100533),new n.Ar(95,158.926385075),new n.Ar(96,159.927050616),new n.Ar(97,160.929665688),new n.Ar(98,161.930981211),new n.Ar(99,162.93399),new n.Ar(100,163.93586),new n.Ar(101,164.93938),new n.Ar(102,165.9416),new n.Ar(103,166.94557),new n.Ar(104,167.94836),new n.Ar(105,168.95287)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(73,137.95287),new n.Ar(74,138.94803),new n.Ar(75,139.945367985),new n.Ar(76,140.94116),new n.Ar(77,141.939073781),new n.Ar(78,142.93475),new n.Ar(79,143.93253),new n.Ar(80,144.92888),new n.Ar(81,145.927180629),new n.Ar(82,146.924037176),new n.Ar(83,147.924298636),new n.Ar(84,148.92324163),new n.Ar(85,149.923654158),new n.Ar(86,150.923098169),new n.Ar(87,151.924071324),new n.Ar(88,152.923430858),new n.Ar(89,153.924686236),new n.Ar(90,154.923500411),new n.Ar(91,155.924743749),new n.Ar(92,156.924021155),new n.Ar(93,157.92541026),new n.Ar(94,158.925343135),new n.Ar(95,159.927164021),new n.Ar(96,160.927566289),new n.Ar(97,161.929484803),new n.Ar(98,162.930643942),new n.Ar(99,163.933347253),new n.Ar(100,164.93488),new n.Ar(101,165.93805),new n.Ar(102,166.94005),new n.Ar(103,167.94364),new n.Ar(104,168.94622),new n.Ar(105,169.95025),new n.Ar(106,170.9533)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(74,139.95379),new n.Ar(75,140.95119),new n.Ar(76,141.946695946),new n.Ar(77,142.94383),new n.Ar(78,143.93907),new n.Ar(79,144.936717),new n.Ar(80,145.932720118),new n.Ar(81,146.930878496),new n.Ar(82,147.927177882),new n.Ar(83,148.927333981),new n.Ar(84,149.925579728),new n.Ar(85,150.92617963),new n.Ar(86,151.924713874),new n.Ar(87,152.925760865),new n.Ar(88,153.924422046),new n.Ar(89,154.92574895),new n.Ar(90,155.924278273),new n.Ar(91,156.925461256),new n.Ar(92,157.924404637),new n.Ar(93,158.92573566),new n.Ar(94,159.925193718),new n.Ar(95,160.926929595),new n.Ar(96,161.926794731),new n.Ar(97,162.928727532),new n.Ar(98,163.929171165),new n.Ar(99,164.931699828),new n.Ar(100,165.932803241),new n.Ar(101,166.935649025),new n.Ar(102,167.93723),new n.Ar(103,168.940303648),new n.Ar(104,169.94267),new n.Ar(105,170.94648),new n.Ar(106,171.94911),new n.Ar(107,172.95344)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(75,141.95986),new n.Ar(76,142.95469),new n.Ar(77,143.95164),new n.Ar(78,144.94688),new n.Ar(79,145.9441),new n.Ar(80,146.93984),new n.Ar(81,147.937269),new n.Ar(82,148.933789944),new n.Ar(83,149.932760914),new n.Ar(84,150.931680791),new n.Ar(85,151.931740598),new n.Ar(86,152.930194506),new n.Ar(87,153.930596268),new n.Ar(88,154.929079084),new n.Ar(89,155.929001869),new n.Ar(90,156.928188059),new n.Ar(91,157.92894573),new n.Ar(92,158.927708537),new n.Ar(93,159.928725679),new n.Ar(94,160.927851662),new n.Ar(95,161.92909242),new n.Ar(96,162.928730286),new n.Ar(97,163.930230577),new n.Ar(98,164.930319169),new n.Ar(99,165.932281267),new n.Ar(100,166.933126195),new n.Ar(101,167.935496424),new n.Ar(102,168.936868306),new n.Ar(103,169.939614951),new n.Ar(104,170.941461227),new n.Ar(105,171.94482),new n.Ar(106,172.94729),new n.Ar(107,173.95115),new n.Ar(108,174.95405)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(76,143.96059),new n.Ar(77,144.95746),new n.Ar(78,145.95212),new n.Ar(79,146.94931),new n.Ar(80,147.94444),new n.Ar(81,148.942780527),new n.Ar(82,149.937171034),new n.Ar(83,150.93746),new n.Ar(84,151.935078452),new n.Ar(85,152.935093125),new n.Ar(86,153.932777294),new n.Ar(87,154.933204273),new n.Ar(88,155.931015001),new n.Ar(89,156.931945517),new n.Ar(90,157.929912),new n.Ar(91,158.930680718),new n.Ar(92,159.929078924),new n.Ar(93,160.930001348),new n.Ar(94,161.928774923),new n.Ar(95,162.930029273),new n.Ar(96,163.929196996),new n.Ar(97,164.9307228),new n.Ar(98,165.93028997),new n.Ar(99,166.932045448),new n.Ar(100,167.932367781),new n.Ar(101,168.934588082),new n.Ar(102,169.935460334),new n.Ar(103,170.938025885),new n.Ar(104,171.939352149),new n.Ar(105,172.9424),new n.Ar(106,173.94434),new n.Ar(107,174.94793),new n.Ar(108,175.95029),new n.Ar(109,176.95437)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(77,145.966495),new n.Ar(78,146.961081),new n.Ar(79,147.95755),new n.Ar(80,148.95265),new n.Ar(81,149.94967),new n.Ar(82,150.944842),new n.Ar(83,151.9443),new n.Ar(84,152.942027631),new n.Ar(85,153.940832325),new n.Ar(86,154.939191562),new n.Ar(87,155.939006895),new n.Ar(88,156.936756069),new n.Ar(89,157.936996),new n.Ar(90,158.934808966),new n.Ar(91,159.935090772),new n.Ar(92,160.933398042),new n.Ar(93,161.933970147),new n.Ar(94,162.932647648),new n.Ar(95,163.933450972),new n.Ar(96,164.932432463),new n.Ar(97,165.933553133),new n.Ar(98,166.932848844),new n.Ar(99,167.934170375),new n.Ar(100,168.934211117),new n.Ar(101,169.935797877),new n.Ar(102,170.936425817),new n.Ar(103,171.938396118),new n.Ar(104,172.939600336),new n.Ar(105,173.942164618),new n.Ar(106,174.943832897),new n.Ar(107,175.946991412),new n.Ar(108,176.94904),new n.Ar(109,177.95264),new n.Ar(110,178.95534)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(78,147.96676),new n.Ar(79,148.96348),new n.Ar(80,149.95799),new n.Ar(81,150.954657965),new n.Ar(82,151.950167),new n.Ar(83,152.94921),new n.Ar(84,153.945651145),new n.Ar(85,154.945792),new n.Ar(86,155.942847109),new n.Ar(87,156.94265865),new n.Ar(88,157.939857897),new n.Ar(89,158.940153735),new n.Ar(90,159.93756),new n.Ar(91,160.937357719),new n.Ar(92,161.93575),new n.Ar(93,162.936265492),new n.Ar(94,163.93452),new n.Ar(95,164.935397592),new n.Ar(96,165.933879623),new n.Ar(97,166.934946862),new n.Ar(98,167.933894465),new n.Ar(99,168.93518712),new n.Ar(100,169.934758652),new n.Ar(101,170.936322297),new n.Ar(102,171.936377696),new n.Ar(103,172.938206756),new n.Ar(104,173.938858101),new n.Ar(105,174.941272494),new n.Ar(106,175.942568409),new n.Ar(107,176.945257126),new n.Ar(108,177.946643396),new n.Ar(109,178.95017),new n.Ar(110,179.95233),new n.Ar(111,180.95615)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(79,149.972668),new n.Ar(80,150.967147),new n.Ar(81,151.96361),new n.Ar(82,152.95869),new n.Ar(83,153.9571),new n.Ar(84,154.953641324),new n.Ar(85,155.952907),new n.Ar(86,156.950101536),new n.Ar(87,157.948577981),new n.Ar(88,158.946615113),new n.Ar(89,159.945383),new n.Ar(90,160.943047504),new n.Ar(91,161.943222),new n.Ar(92,162.941203796),new n.Ar(93,163.941215),new n.Ar(94,164.939605886),new n.Ar(95,165.939762646),new n.Ar(96,166.938307056),new n.Ar(97,167.938698576),new n.Ar(98,168.937648757),new n.Ar(99,169.93847219),new n.Ar(100,170.937909903),new n.Ar(101,171.939082239),new n.Ar(102,172.938926901),new n.Ar(103,173.940333522),new n.Ar(104,174.940767904),new n.Ar(105,175.942682399),new n.Ar(106,176.943754987),new n.Ar(107,177.945951366),new n.Ar(108,178.947324216),new n.Ar(109,179.949879968),new n.Ar(110,180.95197),new n.Ar(111,181.95521),new n.Ar(112,182.95757),new n.Ar(113,183.96117)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(82,153.96425),new n.Ar(83,154.96276),new n.Ar(84,155.959247),new n.Ar(85,156.958127),new n.Ar(86,157.95405528),new n.Ar(87,158.954003),new n.Ar(88,159.950713588),new n.Ar(89,160.950330852),new n.Ar(90,161.947202977),new n.Ar(91,162.947057),new n.Ar(92,163.944422),new n.Ar(93,164.94454),new n.Ar(94,165.94225),new n.Ar(95,166.9426),new n.Ar(96,167.94063),new n.Ar(97,168.941158567),new n.Ar(98,169.93965),new n.Ar(99,170.94049),new n.Ar(100,171.93945798),new n.Ar(101,172.94065),new n.Ar(102,173.940040159),new n.Ar(103,174.941502991),new n.Ar(104,175.941401828),new n.Ar(105,176.943220013),new n.Ar(106,177.943697732),new n.Ar(107,178.945815073),new n.Ar(108,179.94654876),new n.Ar(109,180.949099124),new n.Ar(110,181.950552893),new n.Ar(111,182.953531012),new n.Ar(112,183.95544788),new n.Ar(113,184.95878),new n.Ar(114,185.96092)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(83,155.971689),new n.Ar(84,156.968145),new n.Ar(85,157.966368),new n.Ar(86,158.96232309),new n.Ar(87,159.961358),new n.Ar(88,160.958372992),new n.Ar(89,161.956556553),new n.Ar(90,162.95431665),new n.Ar(91,163.95357),new n.Ar(92,164.950817),new n.Ar(93,165.95047),new n.Ar(94,166.948639),new n.Ar(95,167.947787),new n.Ar(96,168.94592),new n.Ar(97,169.94609),new n.Ar(98,170.94446),new n.Ar(99,171.944739818),new n.Ar(100,172.94459),new n.Ar(101,173.944167937),new n.Ar(102,174.94365),new n.Ar(103,175.944740551),new n.Ar(104,176.944471766),new n.Ar(105,177.945750349),new n.Ar(106,178.945934113),new n.Ar(107,179.947465655),new n.Ar(108,180.947996346),new n.Ar(109,181.950152414),new n.Ar(110,182.951373188),new n.Ar(111,183.954009331),new n.Ar(112,184.955559086),new n.Ar(113,185.9585501),new n.Ar(114,186.96041),new n.Ar(115,187.96371)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(84,157.973939),new n.Ar(85,158.97228),new n.Ar(86,159.968369),new n.Ar(87,160.967089),new n.Ar(88,161.962750303),new n.Ar(89,162.962532),new n.Ar(90,163.95898381),new n.Ar(91,164.958335962),new n.Ar(92,165.955019896),new n.Ar(93,166.954672),new n.Ar(94,167.951863),new n.Ar(95,168.951759),new n.Ar(96,169.948473988),new n.Ar(97,170.94946),new n.Ar(98,171.948228837),new n.Ar(99,172.948884),new n.Ar(100,173.94616),new n.Ar(101,174.94677),new n.Ar(102,175.94559),new n.Ar(103,176.94662),new n.Ar(104,177.945848364),new n.Ar(105,178.947071733),new n.Ar(106,179.946705734),new n.Ar(107,180.948198054),new n.Ar(108,181.948205519),new n.Ar(109,182.950224458),new n.Ar(110,183.950932553),new n.Ar(111,184.953420586),new n.Ar(112,185.954362204),new n.Ar(113,186.957158365),new n.Ar(114,187.958486954),new n.Ar(115,188.96191222),new n.Ar(116,189.963179541)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(85,159.981485),new n.Ar(86,160.977661),new n.Ar(87,161.975707),new n.Ar(88,162.971375872),new n.Ar(89,163.970319),new n.Ar(90,164.967050268),new n.Ar(91,165.965211372),new n.Ar(92,166.962564),new n.Ar(93,167.961609),new n.Ar(94,168.95883),new n.Ar(95,169.958163),new n.Ar(96,170.955547),new n.Ar(97,171.955285),new n.Ar(98,172.953062),new n.Ar(99,173.952114),new n.Ar(100,174.951393),new n.Ar(101,175.95157),new n.Ar(102,176.95027),new n.Ar(103,177.950851081),new n.Ar(104,178.949981038),new n.Ar(105,179.95078768),new n.Ar(106,180.950064596),new n.Ar(107,181.951211444),new n.Ar(108,182.950821349),new n.Ar(109,183.952524289),new n.Ar(110,184.952955747),new n.Ar(111,185.954986529),new n.Ar(112,186.955750787),new n.Ar(113,187.958112287),new n.Ar(114,188.959228359),new n.Ar(115,189.961816139),new n.Ar(116,190.963123592),new n.Ar(117,191.96596)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(86,161.983819),new n.Ar(87,162.982048),new n.Ar(88,163.977927),new n.Ar(89,164.976475),new n.Ar(90,165.971934911),new n.Ar(91,166.971554),new n.Ar(92,167.967832911),new n.Ar(93,168.967076205),new n.Ar(94,169.963569716),new n.Ar(95,170.96304),new n.Ar(96,171.960078),new n.Ar(97,172.959791),new n.Ar(98,173.956307704),new n.Ar(99,174.95708),new n.Ar(100,175.953757941),new n.Ar(101,176.955045),new n.Ar(102,177.953348225),new n.Ar(103,178.953951),new n.Ar(104,179.952308241),new n.Ar(105,180.953274494),new n.Ar(106,181.952186222),new n.Ar(107,182.95311),new n.Ar(108,183.952490808),new n.Ar(109,184.954043023),new n.Ar(110,185.953838355),new n.Ar(111,186.955747928),new n.Ar(112,187.955835993),new n.Ar(113,188.958144866),new n.Ar(114,189.95844521),new n.Ar(115,190.960927951),new n.Ar(116,191.961479047),new n.Ar(117,192.964148083),new n.Ar(118,193.965179314),new n.Ar(119,194.968123889),new n.Ar(120,195.96962255)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(88,164.98758),new n.Ar(89,165.985506),new n.Ar(90,166.980951577),new n.Ar(91,167.979966),new n.Ar(92,168.976390868),new n.Ar(93,169.974441697),new n.Ar(94,170.971779),new n.Ar(95,171.970643),new n.Ar(96,172.967707),new n.Ar(97,173.966804),new n.Ar(98,174.964279),new n.Ar(99,175.963511),new n.Ar(100,176.96117),new n.Ar(101,177.960084944),new n.Ar(102,178.95915),new n.Ar(103,179.958555615),new n.Ar(104,180.957642156),new n.Ar(105,181.958127689),new n.Ar(106,182.956814),new n.Ar(107,183.957388318),new n.Ar(108,184.95659),new n.Ar(109,185.957951104),new n.Ar(110,186.95736083),new n.Ar(111,187.958851962),new n.Ar(112,188.958716473),new n.Ar(113,189.960592299),new n.Ar(114,190.960591191),new n.Ar(115,191.962602198),new n.Ar(116,192.9629237),new n.Ar(117,193.96507561),new n.Ar(118,194.9659768),new n.Ar(119,195.968379906),new n.Ar(120,196.969636496),new n.Ar(121,197.97228),new n.Ar(122,198.973787159)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(90,167.988035),new n.Ar(91,168.986421),new n.Ar(92,169.981734918),new n.Ar(93,170.981251),new n.Ar(94,171.977376138),new n.Ar(95,172.976499642),new n.Ar(96,173.972811276),new n.Ar(97,174.972276),new n.Ar(98,175.969),new n.Ar(99,176.968453),new n.Ar(100,177.964894223),new n.Ar(101,178.965475),new n.Ar(102,179.962023729),new n.Ar(103,180.963177),new n.Ar(104,181.961267637),new n.Ar(105,182.961729),new n.Ar(106,183.959851685),new n.Ar(107,184.960753782),new n.Ar(108,185.959432346),new n.Ar(109,186.960697),new n.Ar(110,187.959395697),new n.Ar(111,188.9608319),new n.Ar(112,189.959930073),new n.Ar(113,190.961684653),new n.Ar(114,191.961035158),new n.Ar(115,192.962984504),new n.Ar(116,193.962663581),new n.Ar(117,194.964774449),new n.Ar(118,195.964934884),new n.Ar(119,196.967323401),new n.Ar(120,197.967876009),new n.Ar(121,198.970576213),new n.Ar(122,199.971423885),new n.Ar(123,200.974496467),new n.Ar(124,201.97574)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(92,170.991183),new n.Ar(93,171.990109),new n.Ar(94,172.986398138),new n.Ar(95,173.984325861),new n.Ar(96,174.981552),new n.Ar(97,175.980269),new n.Ar(98,176.977215),new n.Ar(99,177.975975),new n.Ar(100,178.973412),new n.Ar(101,179.972396),new n.Ar(102,180.969948),new n.Ar(103,181.968621416),new n.Ar(104,182.96762),new n.Ar(105,183.966776046),new n.Ar(106,184.965806956),new n.Ar(107,185.965997671),new n.Ar(108,186.964562),new n.Ar(109,187.965321662),new n.Ar(110,188.9642243),new n.Ar(111,189.964698757),new n.Ar(112,190.963649239),new n.Ar(113,191.964810107),new n.Ar(114,192.964131745),new n.Ar(115,193.96533889),new n.Ar(116,194.965017928),new n.Ar(117,195.966551315),new n.Ar(118,196.966551609),new n.Ar(119,197.968225244),new n.Ar(120,198.968748016),new n.Ar(121,199.970717886),new n.Ar(122,200.971640839),new n.Ar(123,201.973788431),new n.Ar(124,202.975137256),new n.Ar(125,203.977705),new n.Ar(126,204.97961)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(95,174.991411),new n.Ar(96,175.987413248),new n.Ar(97,176.986336874),new n.Ar(98,177.982476325),new n.Ar(99,178.981783),new n.Ar(100,179.978322),new n.Ar(101,180.977806),new n.Ar(102,181.97393546),new n.Ar(103,182.974561),new n.Ar(104,183.970705219),new n.Ar(105,184.971983),new n.Ar(106,185.969460021),new n.Ar(107,186.969785),new n.Ar(108,187.967511693),new n.Ar(109,188.968733187),new n.Ar(110,189.966958568),new n.Ar(111,190.96706311),new n.Ar(112,191.965921572),new n.Ar(113,192.966644169),new n.Ar(114,193.965381832),new n.Ar(115,194.966638981),new n.Ar(116,195.965814846),new n.Ar(117,196.967195333),new n.Ar(118,197.96675183),new n.Ar(119,198.968262489),new n.Ar(120,199.968308726),new n.Ar(121,200.970285275),new n.Ar(122,201.970625604),new n.Ar(123,202.972857096),new n.Ar(124,203.97347564),new n.Ar(125,204.976056104),new n.Ar(126,205.977498672),new n.Ar(127,206.982577025),new n.Ar(128,207.98594)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(96,176.996881),new n.Ar(97,177.994637),new n.Ar(98,178.991466),new n.Ar(99,179.990194),new n.Ar(100,180.986904),new n.Ar(101,181.98561),new n.Ar(102,182.982697),new n.Ar(103,183.98176),new n.Ar(104,184.9791),new n.Ar(105,185.977549881),new n.Ar(106,186.97617),new n.Ar(107,187.97592),new n.Ar(108,188.974290451),new n.Ar(109,189.974473379),new n.Ar(110,190.972261952),new n.Ar(111,191.972770785),new n.Ar(112,192.970548),new n.Ar(113,193.971053),new n.Ar(114,194.96965),new n.Ar(115,195.970515),new n.Ar(116,196.9695362),new n.Ar(117,197.970466294),new n.Ar(118,198.969813837),new n.Ar(119,199.970945394),new n.Ar(120,200.97080377),new n.Ar(121,201.972090569),new n.Ar(122,202.972329088),new n.Ar(123,203.973848646),new n.Ar(124,204.97441227),new n.Ar(125,205.976095321),new n.Ar(126,206.977407908),new n.Ar(127,207.982004653),new n.Ar(128,208.985349125),new n.Ar(129,209.990065574)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(99,180.996714),new n.Ar(100,181.992676101),new n.Ar(101,182.99193),new n.Ar(102,183.988198),new n.Ar(103,184.98758),new n.Ar(104,185.983485388),new n.Ar(105,186.98403),new n.Ar(106,187.979869108),new n.Ar(107,188.98088),new n.Ar(108,189.978180008),new n.Ar(109,190.9782),new n.Ar(110,191.975719811),new n.Ar(111,192.97608),new n.Ar(112,193.974648056),new n.Ar(113,194.975920279),new n.Ar(114,195.97271),new n.Ar(115,196.97338),new n.Ar(116,197.97198),new n.Ar(117,198.972909384),new n.Ar(118,199.97181556),new n.Ar(119,200.972846589),new n.Ar(120,201.972143786),new n.Ar(121,202.973375491),new n.Ar(122,203.973028761),new n.Ar(123,204.974467112),new n.Ar(124,205.974449002),new n.Ar(125,206.975880605),new n.Ar(126,207.97663585),new n.Ar(127,208.981074801),new n.Ar(128,209.984173129),new n.Ar(129,210.988731474),new n.Ar(130,211.991887495),new n.Ar(131,212.9965),new n.Ar(132,213.999798147)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(102,184.997708),new n.Ar(103,185.99648),new n.Ar(104,186.993458),new n.Ar(105,187.992173),new n.Ar(106,188.989505),new n.Ar(107,189.987520007),new n.Ar(108,190.986053),new n.Ar(109,191.985368),new n.Ar(110,192.983662229),new n.Ar(111,193.983430186),new n.Ar(112,194.98112697),new n.Ar(113,195.981236107),new n.Ar(114,196.978934287),new n.Ar(115,197.979024396),new n.Ar(116,198.977576953),new n.Ar(117,199.978141983),new n.Ar(118,200.976970721),new n.Ar(119,201.977674504),new n.Ar(120,202.976868118),new n.Ar(121,203.977805161),new n.Ar(122,204.977374688),new n.Ar(123,205.978482854),new n.Ar(124,206.978455217),new n.Ar(125,207.979726699),new n.Ar(126,208.980383241),new n.Ar(127,209.984104944),new n.Ar(128,210.987258139),new n.Ar(129,211.991271542),new n.Ar(130,212.994374836),new n.Ar(131,213.998698664),new n.Ar(132,215.001832349),new n.Ar(133,216.006199)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(106,189.994293888),new n.Ar(107,190.994653),new n.Ar(108,191.99033039),new n.Ar(109,192.991102),new n.Ar(110,193.988284107),new n.Ar(111,194.988045),new n.Ar(112,195.985469432),new n.Ar(113,196.985567),new n.Ar(114,197.984024384),new n.Ar(115,198.985044507),new n.Ar(116,199.981735),new n.Ar(117,200.982209),new n.Ar(118,201.980704),new n.Ar(119,202.981412863),new n.Ar(120,203.980307113),new n.Ar(121,204.981165396),new n.Ar(122,205.980465241),new n.Ar(123,206.981578228),new n.Ar(124,207.981231059),new n.Ar(125,208.982415788),new n.Ar(126,209.982857396),new n.Ar(127,210.986636869),new n.Ar(128,211.988851755),new n.Ar(129,212.992842522),new n.Ar(130,213.995185949),new n.Ar(131,214.999414609),new n.Ar(132,216.001905198),new n.Ar(133,217.006253),new n.Ar(134,218.008965773)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(108,193.000188),new n.Ar(109,193.997973),new n.Ar(110,194.996554),new n.Ar(111,195.995702),new n.Ar(112,196.993891293),new n.Ar(113,197.99343368),new n.Ar(114,198.991008569),new n.Ar(115,199.990920883),new n.Ar(116,200.988486908),new n.Ar(117,201.988448629),new n.Ar(118,202.986847216),new n.Ar(119,203.987261559),new n.Ar(120,204.986036352),new n.Ar(121,205.986599242),new n.Ar(122,206.985775861),new n.Ar(123,207.986582508),new n.Ar(124,208.986158678),new n.Ar(125,209.987131308),new n.Ar(126,210.987480806),new n.Ar(127,211.990734657),new n.Ar(128,212.99292115),new n.Ar(129,213.996356412),new n.Ar(130,214.998641245),new n.Ar(131,216.002408839),new n.Ar(132,217.004709619),new n.Ar(133,218.008681458),new n.Ar(134,219.011296478),new n.Ar(135,220.015301),new n.Ar(136,221.01814),new n.Ar(137,222.02233),new n.Ar(138,223.02534)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(110,196.001117268),new n.Ar(111,197.001661),new n.Ar(112,197.998779978),new n.Ar(113,198.998309),new n.Ar(114,199.995634148),new n.Ar(115,200.995535),new n.Ar(116,201.993899382),new n.Ar(117,202.994765192),new n.Ar(118,203.991365),new n.Ar(119,204.991668),new n.Ar(120,205.99016),new n.Ar(121,206.990726826),new n.Ar(122,207.989631237),new n.Ar(123,208.990376634),new n.Ar(124,209.989679862),new n.Ar(125,210.99058541),new n.Ar(126,211.990688899),new n.Ar(127,212.993868354),new n.Ar(128,213.995346275),new n.Ar(129,214.998729195),new n.Ar(130,216.000258153),new n.Ar(131,217.003914555),new n.Ar(132,218.005586315),new n.Ar(133,219.009474831),new n.Ar(134,220.011384149),new n.Ar(135,221.015455),new n.Ar(136,222.017570472),new n.Ar(137,223.02179),new n.Ar(138,224.02409),new n.Ar(139,225.02844),new n.Ar(140,226.03089),new n.Ar(141,227.035407),new n.Ar(142,228.038084)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(113,200.006499),new n.Ar(114,201.00458692),new n.Ar(115,202.00396885),new n.Ar(116,203.001423829),new n.Ar(117,204.001221209),new n.Ar(118,204.998663961),new n.Ar(119,205.998486886),new n.Ar(120,206.996859385),new n.Ar(121,207.997133849),new n.Ar(122,208.995915421),new n.Ar(123,209.996398327),new n.Ar(124,210.995529332),new n.Ar(125,211.996194988),new n.Ar(126,212.996174845),new n.Ar(127,213.99895474),new n.Ar(128,215.000326029),new n.Ar(129,216.003187873),new n.Ar(130,217.004616452),new n.Ar(131,218.007563326),new n.Ar(132,219.009240843),new n.Ar(133,220.012312978),new n.Ar(134,221.014245654),new n.Ar(135,222.017543957),new n.Ar(136,223.019730712),new n.Ar(137,224.023235513),new n.Ar(138,225.025606914),new n.Ar(139,226.029343423),new n.Ar(140,227.031833167),new n.Ar(141,228.034776087),new n.Ar(142,229.038426),new n.Ar(143,230.04251),new n.Ar(144,231.045407),new n.Ar(145,232.049654)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(115,203.00921),new n.Ar(116,204.006434513),new n.Ar(117,205.006187),new n.Ar(118,206.004463814),new n.Ar(119,207.005176607),new n.Ar(120,208.001776),new n.Ar(121,209.001944),new n.Ar(122,210.000446),new n.Ar(123,211.000893996),new n.Ar(124,211.999783492),new n.Ar(125,213.000345847),new n.Ar(126,214.000091141),new n.Ar(127,215.002704195),new n.Ar(128,216.003518402),new n.Ar(129,217.00630601),new n.Ar(130,218.007123948),new n.Ar(131,219.010068787),new n.Ar(132,220.011014669),new n.Ar(133,221.013907762),new n.Ar(134,222.01536182),new n.Ar(135,223.01849714),new n.Ar(136,224.020202004),new n.Ar(137,225.023604463),new n.Ar(138,226.025402555),new n.Ar(139,227.029170677),new n.Ar(140,228.031064101),new n.Ar(141,229.034820309),new n.Ar(142,230.037084774),new n.Ar(143,231.04122),new n.Ar(144,232.043693),new n.Ar(145,233.047995),new n.Ar(146,234.050547)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(118,207.012469754),new n.Ar(119,208.012112949),new n.Ar(120,209.009568736),new n.Ar(121,210.009256802),new n.Ar(122,211.007648196),new n.Ar(123,212.007811441),new n.Ar(124,213.006573689),new n.Ar(125,214.006893072),new n.Ar(126,215.006450832),new n.Ar(127,216.008721268),new n.Ar(128,217.009332676),new n.Ar(129,218.011625045),new n.Ar(130,219.012404918),new n.Ar(131,220.014752105),new n.Ar(132,221.015575746),new n.Ar(133,222.017828852),new n.Ar(134,223.01912603),new n.Ar(135,224.021708435),new n.Ar(136,225.023220576),new n.Ar(137,226.026089848),new n.Ar(138,227.027746979),new n.Ar(139,228.031014825),new n.Ar(140,229.032930871),new n.Ar(141,230.036025144),new n.Ar(142,231.038551503),new n.Ar(143,232.042022474),new n.Ar(144,233.04455),new n.Ar(145,234.04842),new n.Ar(146,235.051102),new n.Ar(147,236.055178)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(120,210.015711883),new n.Ar(121,211.016306912),new n.Ar(122,212.012916),new n.Ar(123,213.012962),new n.Ar(124,214.011451),new n.Ar(125,215.011726597),new n.Ar(126,216.011050963),new n.Ar(127,217.013066169),new n.Ar(128,218.013267744),new n.Ar(129,219.015521253),new n.Ar(130,220.015733126),new n.Ar(131,221.018171499),new n.Ar(132,222.018454131),new n.Ar(133,223.020795153),new n.Ar(134,224.02145925),new n.Ar(135,225.023941441),new n.Ar(136,226.024890681),new n.Ar(137,227.027698859),new n.Ar(138,228.028731348),new n.Ar(139,229.03175534),new n.Ar(140,230.033126574),new n.Ar(141,231.03629706),new n.Ar(142,232.03805036),new n.Ar(143,233.041576923),new n.Ar(144,234.043595497),new n.Ar(145,235.04750442),new n.Ar(146,236.04971),new n.Ar(147,237.053894),new n.Ar(148,238.056243)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(122,213.021183209),new n.Ar(123,214.02073923),new n.Ar(124,215.019097612),new n.Ar(125,216.019109649),new n.Ar(126,217.018288571),new n.Ar(127,218.020007906),new n.Ar(128,219.019880348),new n.Ar(129,220.021876493),new n.Ar(130,221.021863742),new n.Ar(131,222.023726),new n.Ar(132,223.023963748),new n.Ar(133,224.025614854),new n.Ar(134,225.026115172),new n.Ar(135,226.02793275),new n.Ar(136,227.028793151),new n.Ar(137,228.031036942),new n.Ar(138,229.032088601),new n.Ar(139,230.034532562),new n.Ar(140,231.035878898),new n.Ar(141,232.03858172),new n.Ar(142,233.040240235),new n.Ar(143,234.043302325),new n.Ar(144,235.045436759),new n.Ar(145,236.048675176),new n.Ar(146,237.05113943),new n.Ar(147,238.054497046),new n.Ar(148,239.05713),new n.Ar(149,240.06098)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(126,218.023487),new n.Ar(127,219.024915423),new n.Ar(128,220.024712),new n.Ar(129,221.026351),new n.Ar(130,222.02607),new n.Ar(131,223.027722956),new n.Ar(132,224.027590139),new n.Ar(133,225.029384369),new n.Ar(134,226.02933975),new n.Ar(135,227.031140069),new n.Ar(136,228.031366357),new n.Ar(137,229.033496137),new n.Ar(138,230.033927392),new n.Ar(139,231.036289158),new n.Ar(140,232.03714628),new n.Ar(141,233.039628196),new n.Ar(142,234.040945606),new n.Ar(143,235.043923062),new n.Ar(144,236.045561897),new n.Ar(145,237.048723955),new n.Ar(146,238.050782583),new n.Ar(147,239.054287777),new n.Ar(148,240.056585734),new n.Ar(149,241.06033),new n.Ar(150,242.062925)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(132,225.033899689),new n.Ar(133,226.035129),new n.Ar(134,227.034958261),new n.Ar(135,228.03618),new n.Ar(136,229.036246866),new n.Ar(137,230.037812591),new n.Ar(138,231.038233161),new n.Ar(139,232.040099),new n.Ar(140,233.04073235),new n.Ar(141,234.042888556),new n.Ar(142,235.044055876),new n.Ar(143,236.046559724),new n.Ar(144,237.048167253),new n.Ar(145,238.050940464),new n.Ar(146,239.052931399),new n.Ar(147,240.056168828),new n.Ar(148,241.058246266),new n.Ar(149,242.061635),new n.Ar(150,243.064273),new n.Ar(151,244.06785)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(134,228.038727686),new n.Ar(135,229.040138934),new n.Ar(136,230.039645603),new n.Ar(137,231.041258),new n.Ar(138,232.041179445),new n.Ar(139,233.04298757),new n.Ar(140,234.043304681),new n.Ar(141,235.0452815),new n.Ar(142,236.046048088),new n.Ar(143,237.048403774),new n.Ar(144,238.0495534),new n.Ar(145,239.052156519),new n.Ar(146,240.05380746),new n.Ar(147,241.056845291),new n.Ar(148,242.058736847),new n.Ar(149,243.061997013),new n.Ar(150,244.06419765),new n.Ar(151,245.067738657),new n.Ar(152,246.070198429),new n.Ar(153,247.07407)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(136,231.04556),new n.Ar(137,232.04659),new n.Ar(138,233.046472),new n.Ar(139,234.047794),new n.Ar(140,235.048029),new n.Ar(141,236.049569),new n.Ar(142,237.049970748),new n.Ar(143,238.051977839),new n.Ar(144,239.053018481),new n.Ar(145,240.055287826),new n.Ar(146,241.056822944),new n.Ar(147,242.059543039),new n.Ar(148,243.061372686),new n.Ar(149,244.064279429),new n.Ar(150,245.066445398),new n.Ar(151,246.069768438),new n.Ar(152,247.072086),new n.Ar(153,248.075745),new n.Ar(154,249.07848)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(137,233.0508),new n.Ar(138,234.05024),new n.Ar(139,235.051591),new n.Ar(140,236.051405),new n.Ar(141,237.052891),new n.Ar(142,238.053016298),new n.Ar(143,239.054951),new n.Ar(144,240.055519046),new n.Ar(145,241.057646736),new n.Ar(146,242.058829326),new n.Ar(147,243.061382249),new n.Ar(148,244.062746349),new n.Ar(149,245.065485586),new n.Ar(150,246.067217551),new n.Ar(151,247.070346811),new n.Ar(152,248.072342247),new n.Ar(153,249.075947062),new n.Ar(154,250.078350687),new n.Ar(155,251.082277873),new n.Ar(156,252.08487)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(138,235.05658),new n.Ar(139,236.05733),new n.Ar(140,237.057127),new n.Ar(141,238.058266),new n.Ar(142,239.058362),new n.Ar(143,240.059749),new n.Ar(144,241.060223),new n.Ar(145,242.06205),new n.Ar(146,243.06300157),new n.Ar(147,244.065167882),new n.Ar(148,245.066355386),new n.Ar(149,246.068666836),new n.Ar(150,247.070298533),new n.Ar(151,248.07308),new n.Ar(152,249.074979937),new n.Ar(153,250.078310529),new n.Ar(154,251.08075344),new n.Ar(155,252.084303),new n.Ar(156,253.08688),new n.Ar(157,254.0906)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(139,237.06207),new n.Ar(140,238.06141),new n.Ar(141,239.062579),new n.Ar(142,240.062295),new n.Ar(143,241.063716),new n.Ar(144,242.063688713),new n.Ar(145,243.065421),new n.Ar(146,244.06599039),new n.Ar(147,245.068039),new n.Ar(148,246.068798807),new n.Ar(149,247.070992043),new n.Ar(150,248.07217808),new n.Ar(151,249.074846818),new n.Ar(152,250.076399951),new n.Ar(153,251.079580056),new n.Ar(154,252.081619582),new n.Ar(155,253.085126791),new n.Ar(156,254.087316198),new n.Ar(157,255.091039),new n.Ar(158,256.09344)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(141,240.06892),new n.Ar(142,241.068662),new n.Ar(143,242.069699),new n.Ar(144,243.069631),new n.Ar(145,244.070969),new n.Ar(146,245.071317),new n.Ar(147,246.072965),new n.Ar(148,247.07365),new n.Ar(149,248.075458),new n.Ar(150,249.076405),new n.Ar(151,250.078654),new n.Ar(152,251.079983592),new n.Ar(153,252.082972247),new n.Ar(154,253.084817974),new n.Ar(155,254.088016026),new n.Ar(156,255.090266386),new n.Ar(157,256.093592),new n.Ar(158,257.095979)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(142,242.07343),new n.Ar(143,243.07451),new n.Ar(144,244.074077),new n.Ar(145,245.075375),new n.Ar(146,246.075281634),new n.Ar(147,247.076819),new n.Ar(148,248.077184411),new n.Ar(149,249.079024),new n.Ar(150,250.079514759),new n.Ar(151,251.081566467),new n.Ar(152,252.082460071),new n.Ar(153,253.085176259),new n.Ar(154,254.086847795),new n.Ar(155,255.089955466),new n.Ar(156,256.091766522),new n.Ar(157,257.095098635),new n.Ar(158,258.097069),new n.Ar(159,259.100588)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(144,245.081017),new n.Ar(145,246.081933),new n.Ar(146,247.081804),new n.Ar(147,248.082909),new n.Ar(148,249.083002),new n.Ar(149,250.084488),new n.Ar(150,251.084919),new n.Ar(151,252.08663),new n.Ar(152,253.08728),new n.Ar(153,254.089725),new n.Ar(154,255.091075196),new n.Ar(155,256.094052757),new n.Ar(156,257.095534643),new n.Ar(157,258.098425321),new n.Ar(158,259.100503),new n.Ar(159,260.103645)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(147,249.087823),new n.Ar(148,250.087493),new n.Ar(149,251.08896),new n.Ar(150,252.088965909),new n.Ar(151,253.090649),new n.Ar(152,254.090948746),new n.Ar(153,255.093232449),new n.Ar(154,256.094275879),new n.Ar(155,257.096852778),new n.Ar(156,258.0982),new n.Ar(157,259.101024),new n.Ar(158,260.102636),new n.Ar(159,261.105743),new n.Ar(160,262.10752)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(148,251.09436),new n.Ar(149,252.09533),new n.Ar(150,253.095258),new n.Ar(151,254.096587),new n.Ar(152,255.096769),new n.Ar(153,256.098763),new n.Ar(154,257.099606),new n.Ar(155,258.101883),new n.Ar(156,259.10299),new n.Ar(157,260.105572),new n.Ar(158,261.106941),new n.Ar(159,262.109692),new n.Ar(160,263.111394)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(149,253.100679),new n.Ar(150,254.100166),new n.Ar(151,255.101492),new n.Ar(152,256.101179573),new n.Ar(153,257.103072),new n.Ar(154,258.103568),new n.Ar(155,259.105628),new n.Ar(156,260.106434),new n.Ar(157,261.108752),new n.Ar(158,262.109918),new n.Ar(159,263.11254),new n.Ar(160,264.113978)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(150,255.107398),new n.Ar(151,256.10811),new n.Ar(152,257.107858),new n.Ar(153,258.109438),new n.Ar(154,259.109721),new n.Ar(155,260.111427),new n.Ar(156,261.112106),new n.Ar(157,262.114153),new n.Ar(158,263.115078),new n.Ar(159,264.117473),new n.Ar(160,265.118659)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(152,258.113151),new n.Ar(153,259.114652),new n.Ar(154,260.114435447),new n.Ar(155,261.116199),new n.Ar(156,262.116477),new n.Ar(157,263.118313),new n.Ar(158,264.118924),new n.Ar(159,265.121066),new n.Ar(160,266.121928)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(153,260.121803),new n.Ar(154,261.1218),new n.Ar(155,262.123009),new n.Ar(156,263.123146),new n.Ar(157,264.12473),new n.Ar(158,265.125198),new n.Ar(159,266.127009),new n.Ar(160,267.12774)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(155,263.12871),new n.Ar(156,264.128408258),new n.Ar(157,265.130001),new n.Ar(158,266.130042),new n.Ar(159,267.131774),new n.Ar(160,268.132156),new n.Ar(161,269.134114)]),j(F(o.t8,1),Vt,3,0,[new n.Ar(156,265.136567),new n.Ar(157,266.13794),new n.Ar(158,267.137526),new n.Ar(159,268.138816),new n.Ar(160,269.139106),new n.Ar(161,270.140723),new n.Ar(162,271.141229)])]);},n.Dr=function(t,A){n.Cr();var e,r;for(r=A-t,e=0;e<n.Br[t].length;e++)if(n.Br[t][e].b==r)return n.Br[t][e].a;return NaN;},n.Er=function(t,A){var e;if(t.b.length!=A.b.length)return t.b.length<A.b.length?-1:1;for(e=0;e<t.b.length;e++)if(t.b[e]!=A.b[e])return t.b[e]<A.b[e]?-1:1;return 0;},n.Fr=function(t){var A,e,r,P,i,D;for(e=0,P=t.a,i=0,D=P.length;i<D;++i)r=P[i],r&&++e;for(t.b=B(o.Q7,H,6,e,15,1),e=0,A=0;A<t.a.length;A++)t.a[A]&&(t.b[e++]=A);},n.Gr=function(t,A){var e;for(n.Fr(t),e=0;e<t.b.length;e++)if(A[t.b[e]])return  true;return  false;},n.Hr=function(t){this.a=B(o.vfb,Tt,6,t,16,1);},x(239,1,{239:1,32:1},n.Hr),s.Sb=function(t){return n.Er(this,t);},o.u8=N(239),n.Mr=function(){n.Mr=f1,n.Lr=j(F(o.O7,1),Ft,6,15,[0,1.00794,4.0026,6.941,9.0122,10.811,12.011,14.007,15.999,18.998,20.18,22.99,24.305,26.982,28.086,30.974,32.066,35.453,39.948,39.098,40.078,44.956,47.867,50.942,51.996,54.938,55.845,58.933,58.693,63.546,65.39,69.723,72.61,74.922,78.96,79.904,83.8,85.468,87.62,88.906,91.224,92.906,95.94,98.906,101.07,102.91,106.42,107.87,112.41,114.82,118.71,121.76,127.6,126.9,131.29,132.91,137.33,138.91,140.12,140.91,144.24,146.92,150.36,151.96,157.25,158.93,162.5,164.93,167.26,168.93,173.04,174.97,178.49,180.95,183.84,186.21,190.23,192.22,195.08,196.97,200.59,204.38,207.2,208.98,209.98,209.99,222.02,223.02,226.03,227.03,232.04,231.04,238.03,237.05,239.05,241.06,244.06,249.08,252.08,252.08,257.1,258.1,259.1,262.11,267.12,268.13,271.13,270.13,277.15,276.15,nr,nr,283.17,285.18,er,er,293.2,rr,rr,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.0141,3.016,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71.0787,156.18828,114.10364,115.0877,103.1447,128.13052,129.11458,57.05182,137.14158,113.15934,113.15934,128.17428,131.19846,147.17646,97.11658,87.0777,101.10458,186.2134,163.17546,99.13246]),n.Jr=j(F(o.O7,1),Ft,6,15,[0,1.007825,4.0026,7.016003,9.012182,11.009305,12,14.003074,15.994915,18.998403,19.992435,22.989767,23.985042,26.98153,27.976927,30.973762,31.97207,34.968852,39.962384,38.963707,39.962591,44.95591,47.947947,50.943962,51.940509,54.938047,55.934939,58.933198,57.935346,62.939598,63.929145,68.92558,73.921177,74.921594,79.91652,78.918336,83.911507,84.911794,87.905619,88.905849,89.904703,92.906377,97.905406,89.92381,101.904348,102.9055,105.903478,106.905092,113.903357,114.90388,119.9022,120.903821,129.906229,126.904473,131.904144,132.905429,137.905232,138.906346,139.905433,140.907647,141.907719,135.92398,151.919729,152.921225,157.924099,158.925342,163.929171,164.930319,165.93029,168.934212,173.938859,174.94077,179.946545,180.947992,183.950928,186.955744,191.961467,192.962917,194.964766,196.966543,201.970617,204.974401,207.976627,208.980374,193.98818,195.99573,199.9957,201.00411,206.0038,210.00923,232.038054,216.01896,238.050784,229.03623,232.041169,237.05005,238.05302,242.06194,240.06228,243.06947,243.07446,248.08275,251.08887,253.09515,257.10295,257.10777,271.13,270.13,277.15,276.15,nr,nr,283.17,285.18,er,er,291.2,rr,rr,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.014,3.01605,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]),n.Kr=j(F(o.Q7,1),H,6,15,[6,1,7,8]);},n.Nr=function(t){var A,e;for(e=t.a,A=0;A<t.b.length;A++)e+=t.b[A]*n.Jr[t.c[A]];return e;},n.Or=function(t){var A,e;for(A=new h.Ckb(),e=0;e<t.b.length;e++)h.Bkb(A,(n.Rk(),n.Lk)[t.c[e]]),t.b[e]>1&&h.Akb(A,t.b[e]);return A.a;},n.Pr=function(t){var A,e;for(e=t.d,A=0;A<t.b.length;A++)e+=t.b[A]*n.Lr[t.c[A]];return e;},n.Qr=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(n.bw(t,1),P=B(o.Q7,H,6,191,15,1),e=0;e<t.q;e++)switch(t.C[e]){case 171:P[1]+=5,P[6]+=3,P[7]+=1,P[8]+=1;break;case 172:P[1]+=12,P[6]+=6,P[7]+=4,P[8]+=1;break;case 173:P[1]+=6,P[6]+=4,P[7]+=2,P[8]+=2;break;case 174:P[1]+=5,P[6]+=4,P[7]+=1,P[8]+=3;break;case 175:P[1]+=5,P[6]+=3,P[7]+=1,P[8]+=1,P[16]+=1;break;case 176:P[1]+=8,P[6]+=5,P[7]+=2,P[8]+=2;break;case 177:P[1]+=7,P[6]+=5,P[7]+=1,P[8]+=3;break;case 178:P[1]+=3,P[6]+=2,P[7]+=1,P[8]+=1;break;case 179:P[1]+=7,P[6]+=6,P[7]+=3,P[8]+=1;break;case 181:case 180:P[1]+=11,P[6]+=6,P[7]+=1,P[8]+=1;break;case 182:P[1]+=12,P[6]+=6,P[7]+=2,P[8]+=1;break;case 183:P[1]+=9,P[6]+=5,P[7]+=1,P[8]+=1,P[16]+=1;break;case 184:P[1]+=9,P[6]+=9,P[7]+=1,P[8]+=1;break;case 185:P[1]+=7,P[6]+=5,P[7]+=1,P[8]+=1;break;case 186:P[1]+=5,P[6]+=3,P[7]+=1,P[8]+=2;break;case 187:P[1]+=7,P[6]+=4,P[7]+=1,P[8]+=2;break;case 188:P[1]+=10,P[6]+=11,P[7]+=2,P[8]+=1;break;case 189:P[1]+=9,P[6]+=9,P[7]+=1,P[8]+=2;break;case 190:P[1]+=9,P[6]+=5,P[7]+=1,P[8]+=1;break;case 1:switch(t.A[e]){case 0:case 1:++P[1];break;case 2:++P[151];break;case 3:++P[152];}break;default:++P[t.C[e]];}for(r=0;r<t.q;r++)t.C[r]>=171&&t.C[r]<=190?P[1]+=2-n.vp(t,r):P[1]+=n.op(t,r);for(u=0,l=1;l<=190;l++)P[l]!=0&&++u;for(this.b=B(o.Q7,H,6,u,15,1),this.c=B(o.Q7,H,6,u,15,1),u=0,Q=0;Q<n.Kr.length;Q++)P[n.Kr[Q]]!=0&&(this.b[u]=P[n.Kr[Q]],this.c[u]=n.Kr[Q],++u,P[n.Kr[Q]]=0);for(;;){for(y="zzz",L=-1,D=1;D<=190;D++)P[D]>0&&h.Wjb(y,(n.Rk(),n.Lk)[D])>0&&(y=(n.Rk(),n.Lk)[D],L=D);if(L==-1)break;this.b[u]=P[L],this.c[u]=L,++u,P[L]=0;}for(this.a=0,this.d=0,A=0;A<t.f;A++)t.C[A]!=1&&t.A[A]!=0&&(D=t.C[A],i=t.A[A],this.a+=n.Dr(D,i)-n.Jr[D],this.d+=n.Dr(D,i)-n.Lr[D]);},x(153,1,{153:1}),s.mb=function(t){var A;if(t===this)return  true;if(!Zt(t,153))return  false;for(A=0;A<this.b.length;A++)if(this.b[A]!=t.b[A])return  false;return  true;},s.a=0,s.d=0,o.v8=N(153),n.Sr=function(t){switch(t){case 6:return 1;case 53:return 2;case 33:return 3;case 34:return 4;case 35:return 5;case 15:return 6;case 16:return 7;case 17:return 8;case 7:return 9;case 8:return 10;case 9:return 11;}return 0;},n.Tr=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O;for(n.bw(t,7),K=0,c=B(o.vfb,Tt,6,t.f,16,1),O=B(o.Q7,H,6,t.f,15,1),i=0;i<t.g;i++)e=t.D[0][i],r=t.D[1][i],D=t.s[e],u=t.s[r],D!=0&&u!=0&&D<0^u<0&&(c[e]=true,c[r]=true),O[e]+=n.fm(t,i),O[r]+=n.fm(t,i);for(Q=0;Q<t.f;Q++)if(K+=t.s[Q],t.s[Q]==1){if(t.C[Q]==7&&!c[Q]){if(O[Q]<=3)K-=1,t.s[Q]=0,t.T=0,t.j[Q]!=t.e[Q]&&(n.ql(t,n.Zo(t,Q,t.e[Q]-1)),n.bw(t,7));else if(Q<t.f&&n.ot(t.p,Q)){for(v=B(o.vfb,Tt,6,t.f,16,1),S=B(o.vfb,Tt,6,t.g,16,1),n.Io(t,Q,true,v,S),L=0;L<t.f;L++)if(v[L]&&t.C[L]==7&&t.s[L]==0&&O[L]==2&&n.Wr(t,S,Q,L)){K-=1;break;}}}}else if(t.s[Q]<0&&(t.C[Q]==6||t.C[Q]==7||t.C[Q]==8||t.C[Q]==16))if(c[Q]){for(q=B(o.Q7,H,6,16,15,1),q[0]=Q,G=1,V=0;V<G;){for(P=0;P<t.r;P++)if(A=-1,t.D[0][P]==q[V]?A=t.D[1][P]:t.D[1][P]==q[V]&&(A=t.D[0][P]),A!=-1&&t.s[A]!=0){for(y=false,g=0;g<G;g++)if(A==q[g]){y=true;break;}y||(G==q.length&&(l=B(o.Q7,H,6,2*q.length,15,1),h.Qkb(q,0,l,0,q.length),q=l),q[G++]=A);}++V;}for(T=0,k=0;k<G;k++)T+=n.Bl(t,q[k]);if(T<0){for(p=-1,E=99,b=0;b<G;b++)n.Bl(t,q[b])<0&&E>n.Sr(n.Tl(t,q[b]))&&(E=n.Sr(n.Tl(t,q[b])),p=q[b]);p!=-1&&(K-=t.s[p],t.s[p]=0,t.T=0);}}else K-=t.s[Q],t.s[Q]=0,t.T=0;return K>0&&(K=n.Ur(t,K)),K;},n.Ur=function(t,A){var e,r,P,i;if(A>0&&(A=n.Xr(t,A,9)),A>0&&(A=n.Xr(t,A,17)),A>0&&(A=n.Xr(t,A,35)),A>0&&(A=n.Xr(t,A,53)),A>0){for(e=0;e<t.f;e++)if(t.s[e]>0){for(P=false,i=0;i<t.j[e];i++)if(r=t.i[e][i],t.s[r]==0&&n.Hm(t,r)&&n.op(t,r)>0){if(t.s[r]=-1,t.T=0,--A,A==0)return 0;P=true;break;}if(P)continue;}}return A>0&&(A=n.Vr(t,A,8,16)),A>0&&(A=n.Vr(t,A,8,15)),A>0&&(A=n.Vr(t,A,8,6)),A>0&&(A=n.Vr(t,A,7,16)),A;},n.Vr=function(t,A,e,r){var P,i,D,u,Q,l,L,y;for(P=0;P<t.f;P++)if(t.C[P]==e&&t.s[P]==0&&n.op(t,P)>0)for(u=false,l=0;l<t.j[P];l++){if(t.n[P][l]==1&&(i=t.i[P][l],t.C[i]==r)){for(y=false,Q=false,L=0;L<t.j[i];L++){if(D=t.i[i][L],t.s[D]<0){Q=true;break;}D!=P&&t.C[D]==8&&t.n[i][L]==2&&(y=true);}if(!Q&&y){if(t.s[P]=-1,t.T=0,--A,A==0)return 0;u=true;}}if(u)break;}return A;},n.Wr=function(t,A,e,r){var P,i,D,u,Q,l,L,y;for(y=B(o.Q7,H,6,t.g,15,1),Q=B(o.vfb,Tt,6,t.g,16,1),D=0;D<t.g;D++)y[D]=t.H[D],A[D]&&(Q[D]=true,t.H[D]=1,t.T=0);for(P=0;P<t.f;P++)if(P!=r&&t.o[P]==0&&Q[t.k[r][0]])for(L=0;L<t.j[P];L++)u=t.k[P][L],t.H[u]=1,t.T=0,Q[u]=false;for(l=0;l<t.j[e];l++)u=t.k[e][l],t.H[u]=1,t.T=0,Q[u]=false;if(n.Ge(new n.Ve(t),Q,false))return t.s[e]=0,t.T=0,t.j[r]!=t.e[r]&&n.ql(t,n.Zo(t,r,t.e[r]-1)),n.bw(t,7),true;for(i=0;i<t.g;i++)n.Mn(t,i,y[i]);return  false;},n.Xr=function(t,A,e){var r;for(r=0;r<t.f;r++)if(t.C[r]==e&&t.s[r]==0&&t.j[r]==0&&(t.s[r]=-1,t.T=0,--A,A==0))return 0;return A;},n.Yr=function(t,A){var e,r;for(r=o0.Dnb(t.b,A),e=w.SAb(r).length;e<10;e++)h.Ekb(t.a,32);h.Ikb(t.a,r);},n.Zr=function(t,A){var e,r,P;if(A<0||A>999){h.Ikb(t.a,"  ?");return;}for(e=false,r=0;r<3;r++)P=A/100|0,P==0?r==2||e?h.Ekb(t.a,48):h.Ekb(t.a,32):(h.Ekb(t.a,48+P&e1),e=true),A=10*(A%100);},n.$r=function(t,A){Rt.shb(A,t.a.a);},n._r=function(t){n.as.call(this,t);},n.as=function(t){n.cs.call(this,t,new h.Lkb());},n.bs=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt,c1,D1,St,gt,w1,I1,L1;for(this.b=new o0.Inb((ct=Pr,ct)),Nt=(h.Pkb(),d1),n.bw(t,15),bt=true,P=0;P<t.f;P++)if((t.u[P]&3)!=0&&(t.u[P]&3)!=3&&(t.u[P]&l1)>>18!=1){bt=false;break;}if(Et=-1,bt){for(z=B(o.Q7,H,6,32,15,1),Ht=0,i=0;i<t.f;i++)if((t.u[i]&3)!=0&&(t.u[i]&3)!=3&&(t.u[i]&l1)>>18==1){lt=(t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&F0)>>20,++z[lt],Ht<z[lt]&&(Ht=z[lt],Et=lt);break;}}for(this.a=e||new h.Kkb(),xt=t.P!=null?t.P:"",h.Ikb(this.a,xt+Nt),h.Ikb(this.a,"Actelion Java MolfileCreator 1.0"+Nt+Nt),n.Zr(this,t.q),n.Zr(this,t.r),h.Ikb(this.a,"  0  0"),n.Zr(this,bt?0:1),h.Ikb(this.a,"  0  0  0  0  0999 V2000"+Nt),ut=t.q==1,l=1;l<t.q;l++)if(t.J[l].a!=t.J[0].a||t.J[l].b!=t.J[0].b||t.J[l].c!=t.J[0].c){ut=true;break;}if(it=1,ut)if(A!=0)it=A;else if(p=n.Wl(t,t.q,t.r,(n.Rk(),n.Qk)),p!=0)(p<1||p>3)&&(it=1.5/p);else {for(Kt=s0,i=1;i<t.q;i++)for(Q=0;Q<i;Q++)O=t.J[Q].a-t.J[i].a,tt=t.J[Q].b-t.J[i].b,Qt=t.J[Q].c-t.J[i].c,K=O*O+tt*tt+Qt*Qt,Kt>K&&(Kt=K);it=3/Kt;}for(L=0;L<t.q;L++)ut?(n.Yr(this,it*t.J[L].a),n.Yr(this,it*-t.J[L].b),n.Yr(this,it*-t.J[L].c)):h.Ikb(this.a,"    0.0000    0.0000    0.0000"),(t.v==null?null:t.v[L])!=null?h.Ikb(this.a," L  "):pt(J(t.B[L],1),0)?h.Ikb(this.a," A  "):t.C[L]>=129&&t.C[L]<=144||t.C[L]==154?h.Ikb(this.a," R# "):(k=(n.Rk(),n.Lk)[t.C[L]],h.Ikb(this.a," "+k),w.SAb(k).length==1?h.Ikb(this.a,"  "):w.SAb(k).length==2&&h.Ikb(this.a," ")),h.Ikb(this.a," 0  0  0"),Dt=J(X0,t.B[L]),nt(Dt,0)==0?h.Ikb(this.a,"  0"):nt(Dt,384)==0?h.Ikb(this.a,"  3"):nt(Dt,128)==0?h.Ikb(this.a,"  2"):nt(Dt,1792)==0?h.Ikb(this.a,"  1"):nt(Dt,1664)==0&&h.Ikb(this.a,"  2"),h.Ikb(this.a,pt(J(t.B[L],L0),0)?"  1":"  0"),L1=((t.u[L]&bA)>>>27)-1,L1==-1?h.Ikb(this.a,"  0"):L1==0?h.Ikb(this.a," 15"):n.Zr(this,L1),h.Ikb(this.a,"  0  0  0"),n.Zr(this,d.Math.abs(t.w[L])),h.Ikb(this.a,"  0  0"+Nt);for(E=0;E<t.r;E++){switch(t.H[E]){case 1:Bt=1,gt=0;break;case 2:Bt=2,gt=0;break;case 4:Bt=3,gt=0;break;case 129:Bt=1,gt=6;break;case 257:Bt=1,gt=1;break;case 386:Bt=2,gt=3;break;case 8:Bt=4,gt=0;break;case 16:Bt=8,gt=0;break;default:Bt=1,gt=0;}bt&&(gt==1||gt==6)&&(i=t.D[0][E],(t.u[i]&l1)>>18==2?gt=0:(t.u[i]&l1)>>18==1&&((t.u[i]&l1)>>18!=1&&(t.u[i]&l1)>>18!=2?-1:(t.u[i]&F0)>>20)!=Et&&(gt=4)),q=t.G[E]&31,q!=0&&(q==8?Bt=4:q==3?Bt=5:q==9?Bt=6:q==10?Bt=7:Bt=8),D1=t.G[E]&384,I1=D1==0?0:D1==256?1:2,n.Zr(this,1+t.D[0][E]),n.Zr(this,1+t.D[1][E]),n.Zr(this,Bt),n.Zr(this,gt),h.Ikb(this.a,"  0"),n.Zr(this,I1),h.Ikb(this.a,"  0"+Nt);}for(kt=0,y=0;y<t.q;y++)t.s[y]!=0&&++kt;if(kt!=0)for(G=0,i=0;i<t.q;i++)t.s[i]!=0&&(G==0&&(h.Ikb(this.a,"M  CHG"),n.Zr(this,d.Math.min(8,kt))),h.Ikb(this.a," "),n.Zr(this,i+1),V=t.s[i],V<0?(h.Ikb(this.a,"  -"),V=-V):h.Ikb(this.a,"   "),h.Ekb(this.a,48+V&e1),--kt,(++G==8||kt==0)&&(G=0,h.Ikb(this.a,Nt)));for(kt=0,T=0;T<t.q;T++)t.A[T]==0||++kt;if(kt!=0)for(G=0,i=0;i<t.q;i++)t.A[i]!=0&&(G==0&&(h.Ikb(this.a,"M  ISO"),n.Zr(this,d.Math.min(8,kt))),h.Ikb(this.a," "),n.Zr(this,i+1),h.Ikb(this.a," "),n.Zr(this,t.A[i]),--kt,(++G==8||kt==0)&&(G=0,h.Ikb(this.a,Nt)));for(kt=0,c=0;c<t.q;c++)(t.u[c]&48)!=0&&++kt;if(kt!=0){for(G=0,i=0;i<t.q;i++)if((t.u[i]&48)!=0){switch(G==0&&(h.Ikb(this.a,"M  RAD"),n.Zr(this,d.Math.min(8,kt))),h.Ikb(this.a," "),n.Zr(this,i+1),t.u[i]&48){case 16:h.Ikb(this.a,"   1");break;case 32:h.Ikb(this.a,"   2");break;case 48:h.Ikb(this.a,"   3");}--kt,(++G==8||kt==0)&&(G=0,h.Ikb(this.a,Nt));}}for(kt=0,b=0;b<t.q;b++)(t.C[b]>=129&&t.C[b]<=144||t.C[b]==154)&&++kt;if(kt!=0)for(G=0,i=0;i<t.q;i++)S=t.C[i],(S>=129&&S<=144||S==154)&&(G==0&&(h.Ikb(this.a,"M  RGP"),n.Zr(this,d.Math.min(8,kt))),h.Ikb(this.a," "),n.Zr(this,i+1),h.Ikb(this.a," "),n.Zr(this,S==154?0:S>=142?S-141:S-125),--kt,(++G==8||kt==0)&&(G=0,h.Ikb(this.a,Nt)));if(t.K){for(kt=0,i=0;i<t.q;i++)pt(J(t.B[i],120),0)&&++kt;if(kt!=0)for(G=0,Q=0;Q<t.q;Q++)c1=J(t.B[Q],120),nt(c1,0)!=0&&(G==0&&(h.Ikb(this.a,"M  RBC"),n.Zr(this,d.Math.min(8,kt))),h.Ikb(this.a," "),n.Zr(this,Q+1),nt(c1,112)==0?h.Ikb(this.a,"  -1"):nt(c1,8)==0?h.Ikb(this.a,"   1"):nt(c1,104)==0?h.Ikb(this.a,"   2"):nt(c1,88)==0?h.Ikb(this.a,"   3"):nt(c1,56)==0&&h.Ikb(this.a,"   4"),--kt,(++G==8||kt==0)&&(G=0,h.Ikb(this.a,Nt)));for(g=0;g<t.q;g++)if(v=t.v==null?null:t.v[g],v!=null){for(h.Ikb(this.a,"M  ALS "),n.Zr(this,g+1),n.Zr(this,v.length),h.Ikb(this.a,pt(J(t.B[g],1),0)?" T ":" F "),ot=0;ot<v.length;ot++)switch(Ut=(n.Rk(),n.Lk)[v[ot]],w.SAb(Ut).length){case 1:h.Ikb(this.a,Ut+"   ");break;case 2:h.Ikb(this.a,Ut+"  ");break;case 3:h.Ikb(this.a,Ut+" ");break;default:h.Ikb(this.a,"   ?");}h.Ikb(this.a,Nt);}for(kt=0,D=0;D<t.q;D++)pt(J(t.B[D],6144),0)&&++kt;if(kt!=0)for(G=0,Q=0;Q<t.q;Q++)w1=J(t.B[Q],6144),nt(w1,0)!=0&&(G==0&&(h.Ikb(this.a,"M  SUB"),n.Zr(this,d.Math.min(8,kt))),h.Ikb(this.a," "),n.Zr(this,Q+1),pt(J(w1,Z1),0)?h.Ikb(this.a,"   "+(t.e[Q]+1)):h.Ikb(this.a,"  -2"),--kt,(++G==8||kt==0)&&(G=0,h.Ikb(this.a,Nt)));}for(kt=0,u=0;u<t.q;u++)(t.t==null||t.t[u]==null?null:h.$jb((It=t.t[u],q1.Cnb(),It)))!=null&&++kt;if(kt!=0)for(St=0,r=0;r<t.q;r++)Ut=t.t==null||t.t[r]==null?null:h.$jb((wt=t.t[r],q1.Cnb(),wt)),Ut!=null&&(++St,h.Ikb(this.a,"M  STY  1 "),n.Zr(this,St),h.Ikb(this.a," DAT"),h.Ikb(this.a,Nt),h.Ikb(this.a,"M  SLB  1 "),n.Zr(this,St),h.Ikb(this.a," "),n.Zr(this,St),h.Ikb(this.a,Nt),h.Ikb(this.a,"M  SAL "),n.Zr(this,St),h.Ikb(this.a,"  1 "),n.Zr(this,r+1),h.Ikb(this.a,Nt),h.Ikb(this.a,"M  SDT "),n.Zr(this,St),h.Ikb(this.a," "),h.Ikb(this.a,ir),h.Ikb(this.a,Nt),h.Ikb(this.a,"M  SDD "),n.Zr(this,St),h.Ikb(this.a," "),n.Yr(this,t.J[r].a),n.Yr(this,t.J[r].b),h.Ikb(this.a,"    DA    ALL  1       5"),h.Ikb(this.a,Nt),h.Ikb(this.a,"M  SED "),n.Zr(this,St),h.Ikb(this.a," "),h.Ikb(this.a,Ut),h.Ikb(this.a,Nt));h.Ikb(this.a,FA+Nt);},n.cs=function(t,A){n.bs.call(this,t,0,A);},x(112,1,{},n._r,n.bs),o.x8=N(112),n.ds=function(t,A,e,r,P,i){var D,u,Q,l;switch(l=1,u=false,P){case 1:l=257;break;case 3:l=386;break;case 4:l=257,u=true;break;case 6:l=129;break;default:switch(r){case 1:l=1;break;case 2:l=2;break;case 3:l=4;break;case 4:l=8;break;case 9:l=16;}}if(D=n.Wk(t.j,A,e,l),Q=0,u&&n.on(t.j,A,1,-1),r>4)switch(r){case 5:Q|=3;break;case 6:Q|=9;break;case 7:Q|=10;break;case 8:l!=16&&(Q|=127);}return i==1&&(Q|=256),i==2&&(Q|=128),Q!=0&&n.Ln(t.j,D,Q,true),D;},n.es=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(D=B(o.Q7,H,6,t.j.q,15,1),e=0;e<t.j.q;e++)D[e]=-n.Bl(t.j,e);for(n.Ge(new n.Ve(t.j),null,true),r=0;r<t.j.q;r++)D[r]+=n.Bl(t.j,r);for(A=0;A<t.j.q;A++)if(D[A]!=0)for(u=-D[A],P=0;P<t.j.r;P++)for(l=0;l<2;l++)u>0&&n.im(t.j,P)==16&&n.Zl(t.j,1-l,P)==A&&(y=n.Zl(t.j,l,P),n.Lm(t.j,y)&&(L=n.gs(t,y),i=n.Bl(t.j,y),i<L&&(Q=d.Math.min(u,L-i),n.jn(t.j,y,i+Q),u-=Q)));},n.fs=function(t,A){return t.j=null,n.As(t,new Rt.ehb(new Rt.phb(A)))?t.j:null;},n.gs=function(t,A){var e,r;return e=n.Tl(t.j,A),r=e<(n.Rk(),n.Nk).length?n.Nk[e]:null,r==null?0:r[r.length-1];},n.hs=function(t,A){var e;return e=t.b?U.eqb(t.b,h.qjb(A)):null,e?e.a:A-1;},n.is=function(t,A){var e;return e=t.c?U.eqb(t.c,h.qjb(A)):null,e?e.a:A-1;},n.js=function(t,A){var e,r;if(A!=null)for(n.bw(t.j,1),e=0;e<t.j.f;e++)A[e]!=0&&(r=n.mm(t.j,e,n.vp(t.j,e)),A[e]==15?r>=0&&n.gn(t.j,e,0):A[e]!=n.om(t.j,e)&&n.gn(t.j,e,A[e]-r));},n.ks=function(t,A){var e,r;for(e=A;e<w.SAb(t).length;e++){if(r=(w.NAb(e,w.SAb(t).length),w.SAb(t).charCodeAt(e)),r==32||r==9)return  -1;if(r==61)return e;}return  -1;},n.ls=function(t,A){var e;if(A==-1)return  -1;for(e=A+1;e<w.SAb(t).length;e++)if(w.NAb(e,w.SAb(t).length),w.SAb(t).charCodeAt(e)!=32&&(w.NAb(e,w.SAb(t).length),w.SAb(t).charCodeAt(e)!=9))return e;return  -1;},n.ms=function(t,A){var e,r,P,i;for(e=false,P=false,r=A;r<w.SAb(t).length;r++)if(i=(w.NAb(r,w.SAb(t).length),w.SAb(t).charCodeAt(r)),i==40?e=true:i==41?e=false:i==34&&(P=!P),!e&&!P&&(i==32||i==9))return r;return  -1;},n.ns=function(t){return w.SAb(t).indexOf("ATOMS=(")!=-1?qn:w.SAb(t).indexOf("BONDS=(")!=-1?"BONDS":null;},n.os=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt;for(g=0,k=(u=n.ms(A,g),u==-1?w.SAb(A).length:u),r=h.Mib((w.MAb(g,k,w.SAb(A).length),w.SAb(A).substr(g,k-g)),r1,qt),g=n.ls(A,k),k=(Q=n.ms(A,g),Q==-1?w.SAb(A).length:Q),S=(w.MAb(g,k,w.SAb(A).length),w.SAb(A).substr(g,k-g)),G=null,P=false,v=n.ws(A),v!=0&&(G=n.ps(A),v<0&&(P=true),k=d.Math.abs(v)),g=n.ls(A,k),k=(l=n.ms(A,g),l==-1?w.SAb(A).length:l),O=x0((w.MAb(g,k,w.SAb(A).length),w.SAb(A).substr(g,k-g))),g=n.ls(A,k),k=(L=n.ms(A,g),L==-1?w.SAb(A).length:L),tt=x0((w.MAb(g,k,w.SAb(A).length),w.SAb(A).substr(g,k-g))),g=n.ls(A,k),k=(y=n.ms(A,g),y==-1?w.SAb(A).length:y),Qt=x0((w.MAb(g,k,w.SAb(A).length),w.SAb(A).substr(g,k-g))),g=n.ls(A,k),k=(T=n.ms(A,g),T==-1?w.SAb(A).length:T),E=h.Mib((w.MAb(g,k,w.SAb(A).length),w.SAb(A).substr(g,k-g)),r1,qt),e=n.Tk(t.j,O,-tt,-Qt),e+1!=r&&(!t.b&&(t.b=new U.Xwb()),U.Pwb(t.b,h.qjb(r),h.qjb(e))),G!=null&&(n.Cn(t.j,e,6),n.qn(t.j,e,G,P)),E!=0&&n.rn(t.j,e,E,false),h.akb(S,"A")||h.akb(S,"*")?n.vn(t.j,e,1,true):h.akb(S,"Q")?(p=B(o.Q7,H,6,1,15,1),p[0]=6,n.qn(t.j,e,p,true)):n.Cn(t.j,e,n.po(S,67));(g=n.ls(A,k))!=-1;)if(k=(D=n.ms(A,g),D==-1?w.SAb(A).length:D),q=(w.MAb(g,k,w.SAb(A).length),w.SAb(A).substr(g,k-g)),b=h.hkb(q,X1(61)),c=(w.MAb(0,b,w.SAb(q).length),w.SAb(q).substr(0,b)),K=h.Mib((w.NAb(b+1,w.SAb(q).length+1),w.SAb(q).substr(b+1)),r1,qt),h.akb(c,"CHG"))n.jn(t.j,e,K);else if(h.akb(c,"RAD"))switch(K){case 1:n.wn(t.j,e,16);break;case 2:n.wn(t.j,e,32);break;case 3:n.wn(t.j,e,48);}else if(!h.akb(c,"CFG")){if(h.akb(c,"MASS"))n.tn(t.j,e,K);else if(h.akb(c,"VAL"))n.gn(t.j,e,K==-1?0:K==0?-1:K);else if(h.akb(c,"HCOUNT"))switch(K){case 0:break;case  -1:n.vn(t.j,e,1792,true);break;case 1:n.vn(t.j,e,128,true);break;case 2:n.vn(t.j,e,384,true);break;default:n.vn(t.j,e,896,true);}else if(h.akb(c,"SUBST")){if(K==-1)n.vn(t.j,e,B1,true);else if(K>0){for(V=0,i=0;i<t.j.r;i++)(n.Zl(t.j,0,i)==e||n.Zl(t.j,1,i)==e)&&++V;K>V&&n.vn(t.j,e,Z1,true);}}else if(h.akb(c,"RBCNT"))switch(K){case 3:case  -1:n.vn(t.j,e,112,true);break;case 1:n.vn(t.j,e,8,true);break;case 2:n.vn(t.j,e,104,true);break;case 4:n.vn(t.j,e,56,true);}}},n.ps=function(t){var A,e,r,P,i,D,u,Q;if(u=null,e=w.SAb(t).indexOf("["),r=w.SAb(t).indexOf("]",e),e>=0&&r>0){for(A=B(o.Q7,H,6,16,15,1),Q=(w.MAb(e+1,r,w.SAb(t).length),w.SAb(t).substr(e+1,r-(e+1))),P=0,D=true;D&&P<16;)e=w.SAb(Q).indexOf(","),i=null,e==-1?(i=Q,D=false):(i=(w.MAb(0,e,w.SAb(Q).length),w.SAb(Q).substr(0,e)),Q=(w.NAb(e+1,w.SAb(Q).length+1),w.SAb(Q).substr(e+1))),A[P++]=n.po(i,1);u=B(o.Q7,H,6,P,15,1),h.Qkb(A,0,u,0,P);}return u;},n.qs=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p;for(b=0,g=(Q=n.ms(A,b),Q==-1?w.SAb(A).length:Q),i=h.Mib((w.MAb(b,g,w.SAb(A).length),w.SAb(A).substr(b,g-b)),r1,qt),b=n.ls(A,g),g=(l=n.ms(A,b),l==-1?w.SAb(A).length:l),D=h.Mib((w.MAb(b,g,w.SAb(A).length),w.SAb(A).substr(b,g-b)),r1,qt),b=n.ls(A,g),g=(L=n.ms(A,b),L==-1?w.SAb(A).length:L),e=n.hs(t,h.Mib((w.MAb(b,g,w.SAb(A).length),w.SAb(A).substr(b,g-b)),r1,qt)),b=n.ls(A,g),g=(y=n.ms(A,b),y==-1?w.SAb(A).length:y),r=n.hs(t,h.Mib((w.MAb(b,g,w.SAb(A).length),w.SAb(A).substr(b,g-b)),r1,qt)),v=0,S=0;(b=n.ls(A,g))!=-1;)if(g=(u=n.ms(A,b),u==-1?w.SAb(A).length:u),k=(w.MAb(b,g,w.SAb(A).length),w.SAb(A).substr(b,g-b)),c=h.hkb(k,X1(61)),T=(w.MAb(0,c,w.SAb(k).length),w.SAb(k).substr(0,c)),p=h.Mib((w.NAb(c+1,w.SAb(k).length+1),w.SAb(k).substr(c+1)),r1,qt),h.akb(T,"CFG"))switch(p){case 1:v=1;break;case 2:v=D==2?3:4;break;case 3:v=6;}else h.akb(T,"TOPO")&&(S=p);P=n.ds(t,e,r,D,v,S),P+1!=i&&(!t.c&&(t.c=new U.Xwb()),U.Pwb(t.c,h.qjb(i),h.qjb(P)));},n.rs=function(t,A){var e,r,P,i,D,u;if(u=n.ns(A),u!=null){if(D=n.ts(A,u),h.akb(w.SAb(A).substr(0,13),"MDLV30/STEABS")){if(h.akb(u,qn))for(i=0;i<D.length;i++)n.on(t.j,n.hs(t,D[i]),0,-1);else for(P=0;P<D.length;P++)n.Gn(t.j,n.is(t,D[P]),0,-1);}else if(h.akb(w.SAb(A).substr(0,13),"MDLV30/STERAC")){if(r=h.Mib(h.rkb(A,13,n.ms(A,13)),r1,qt),h.akb(u,qn))for(i=0;i<D.length;i++)n.on(t.j,n.hs(t,D[i]),1,r-1);else for(P=0;P<D.length;P++)n.Gn(t.j,n.is(t,D[P]),1,r-1);}else if(h.akb(w.SAb(A).substr(0,13),"MDLV30/STEREL")){if(r=h.Mib(h.rkb(A,13,n.ms(A,13)),r1,qt),h.akb(u,qn))for(i=0;i<D.length;i++)n.on(t.j,n.hs(t,D[i]),2,r-1);else for(P=0;P<D.length;P++)n.Gn(t.j,n.is(t,D[P]),2,r-1);}else if(h.akb(w.SAb(A).substr(0,13),"MDLV30/HILITE"))if(h.akb(u,qn))for(P=0;P<D.length;P++)n.kn(t.j,n.hs(t,D[P]),448);else for(P=0;P<D.length;P++)e=n.is(t,D[P]),n.kn(t.j,n.Zl(t.j,0,e),448),n.kn(t.j,n.Zl(t.j,1,e),448);}},n.ss=function(t,A){var e,r,P;t.j||h.akb(w.SAb(A).substr(0,6),"COUNTS")&&(e=n.ls(A,n.ms(A,7)),r=h.Mib(h.rkb(A,7,n.ms(A,7)),r1,qt),P=h.Mib(h.rkb(A,e,n.ms(A,e)),r1,qt),t.j=new n.uw(r,P));},n.ts=function(t,A){var e,r,P,i,D,u;for(i=w.SAb(t).indexOf(A+"=(")+w.SAb(A).length+2,D=h.ikb(t,X1(41),i),P=n.ms(t,i),e=h.Mib((w.MAb(i,P,w.SAb(t).length),w.SAb(t).substr(i,P-i)),r1,qt),u=B(o.Q7,H,6,e,15,1),r=0;r<e;r++)i=n.ls(t,P),P=n.ms(t,i),(P==-1||P>D)&&(P=D),u[r]=h.Mib((w.MAb(i,P,w.SAb(t).length),w.SAb(t).substr(i,P-i)),r1,qt);return u;},n.us=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p;if(T=0,c=(i=n.ms(A,T),i==-1?w.SAb(A).length:i),p=(w.MAb(T,c,w.SAb(A).length),w.SAb(A).substr(T,c-T)),h.akb("DEFAULT",p)&&(T=n.ls(A,c),c=(D=n.ms(A,T),D==-1?w.SAb(A).length:D),p=(w.MAb(T,c,w.SAb(A).length),w.SAb(A).substr(T,c-T)),h.akb(w.SAb(p).substr(0,5),"CLASS")&&(T=n.ls(A,c),c=(u=n.ms(A,T),u==-1?w.SAb(A).length:u),p=(w.MAb(T,c,w.SAb(A).length),w.SAb(A).substr(T,c-T)),h.akb("-",p)&&(T=n.ls(A,c),c=(Q=n.ms(A,T),Q==-1?w.SAb(A).length:Q),p=(w.MAb(T,c,w.SAb(A).length),w.SAb(A).substr(T,c-T))))),h.Mib(p,r1,qt),T=n.ls(A,c),c=(l=n.ms(A,T),l==-1?w.SAb(A).length:l),S=(w.MAb(T,c,w.SAb(A).length),w.SAb(A).substr(T,c-T)),!!h.akb("DAT",S)){for(T=n.ls(A,c),c=(P=n.ms(A,T),P==-1?w.SAb(A).length:P),h.Mib((w.MAb(T,c,w.SAb(A).length),w.SAb(A).substr(T,c-T)),r1,qt),v=new U.Xwb(),T=n.ls(A,c);T!=-1;)T=n.zs(v,A,T);r=(g=qn,U.Xnb(U.Kwb(v,g))),y=(k="FIELDNAME",U.Xnb(U.Kwb(v,k))),L=(b="FIELDDATA",U.Xnb(U.Kwb(v,b))),!(!h.akb(ir,y)||r==null||L==null)&&(e=h.okb(r," "),e.length==2&&n.mn(t.j,h.Mib(e[1],r1,qt)-1,L));}},n.vs=function(t){var A,e,r,P,i,D,u,Q,l;for(Q=B(o.Q7,H,6,t.j.q,15,1),r=0;r<t.j.r;r++)if(n.im(t.j,r)==8)for(D=0;D<2;D++)Q[n.Zl(t.j,D,r)]=1;for(P=0;P<t.j.r;P++)for(l=n.fm(t.j,P),i=0;i<2;i++)Q[n.Zl(t.j,i,P)]+=l;for(e=0;e<t.j.r;e++)if(n.fm(t.j,e)==1){for(i=0;i<2;i++)if(u=n.Zl(t.j,1-i,e),n.Lm(t.j,u)&&(A=n.Zl(t.j,i,e),n.Hm(t.j,A)&&Q[A]>n.om(t.j,A))){n.Mn(t.j,e,16);continue;}}},n.ws=function(t){var A,e;if(w.SAb(t).indexOf("[")>=0){if(A=w.SAb(t).indexOf(" NOT["),e=w.SAb(t).indexOf("]",A),A>=0&&e>0)return -(e+1);if(A=w.SAb(t).indexOf(" ["),e=w.SAb(t).indexOf("]",A),A>=0&&e>0)return e+1;if(A=w.SAb(t).indexOf(" 'NOT["),e=w.SAb(t).indexOf("]'",A),A>=0&&e>0)return -(e+2);if(A=w.SAb(t).indexOf(" '["),e=w.SAb(t).indexOf("]'",A),A>=0&&e>0)return e+2;h.Pkb();}return 0;},n.xs=function(t,A,e){return t.j=A,n.As(t,e);},n.ys=function(t){return w.SAb(t).length==0?0:h.Mib(t,r1,qt);},n.zs=function(t,A,e){var r,P,i,D,u,Q,l;return i=n.ks(A,e),i==-1?-1:(P=(r=n.ms(A,i+1),r==-1?w.SAb(A).length:r),D=(w.MAb(e,i,w.SAb(A).length),w.SAb(A).substr(e,i-e)),l=(w.MAb(i+1,P,w.SAb(A).length),w.SAb(A).substr(i+1,P-(i+1))),(h.akb(w.SAb(l).substr(0,1),"(")&&(Q=w.SAb(")").length,h.akb(w.SAb(l).substr(w.SAb(l).length-Q,Q),")"))||h.akb(w.SAb(l).substr(0,1),'"')&&(u=w.SAb('"').length,h.akb(w.SAb(l).substr(w.SAb(l).length-u,u),'"')))&&(l=h.rkb(l,1,w.SAb(l).length-1)),U.Pwb(t,D,l),n.ls(A,P));},n.As=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt,c1,D1,St,gt,w1,I1;D1=null,b=null;try{if(t.f=null,t.j&&(n._v(t.j),n.On(t.j,!1)),lt=O=Rt.dhb(A),lt==null||(O=Rt.dhb(A))==null||(O=Rt.dhb(A))==null||(t.k=w.SAb(O).indexOf("From CSD data. Using bond type 'Any'")!=-1,t.e=w.SAb(O).indexOf("From CSD data.")!=-1,(O=Rt.dhb(A))==null))return !1;t.g=!1,t.d=t.a;try{ut=h.Mib(h.skb((w.MAb(0,3,w.SAb(O).length),w.SAb(O).substr(0,3))),r1,qt),Dt=h.Mib(h.skb((w.MAb(3,6,w.SAb(O).length),w.SAb(O).substr(3,3))),r1,qt),ot=n.ys(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3)))),t.d=t.d|n.ys(h.skb((w.MAb(12,15,w.SAb(O).length),w.SAb(O).substr(12,3))))==1,t.g=w.SAb(O).length>=39&&h.akb(w.SAb(O).substr(34,w.SAb("V3000").length),"V3000");}catch(L1){if(L1=C1(L1),Zt(L1,19))return !1;throw st(L1);}if(t.g)return ct=n.Bs(t,A),n.Un(t.j,lt),ct;if(!t.j&&(t.j=new n.uw(ut,Dt)),n.Un(t.j,lt),t.d||(t.j.L=!0),ut==0){for(;O!=null&&!(h.akb(O,FA)||h.akb(O,le)||h.akb((w.NAb(1,w.SAb(O).length+1),w.SAb(O).substr(1)),"$"));)O=Rt.dhb(A);return !0;}for(p=0;p<ut;p++){if((O=Rt.dhb(A))==null)return !1;switch(gt=x0(h.skb((w.MAb(0,10,w.SAb(O).length),w.SAb(O).substr(0,10)))),w1=x0(h.skb((w.MAb(10,20,w.SAb(O).length),w.SAb(O).substr(10,10)))),I1=x0(h.skb((w.MAb(20,30,w.SAb(O).length),w.SAb(O).substr(20,10)))),r=n.Tk(t.j,gt,-w1,-I1),K=h.skb((w.MAb(31,34,w.SAb(O).length),w.SAb(O).substr(31,3))),h.akb(K,"A")||h.akb(K,"*")?n.vn(t.j,r,1,!0):h.akb(K,"Q")?(tt=B(o.Q7,H,6,1,15,1),tt[0]=6,n.qn(t.j,r,tt,!0)):(D=n.po(K,67),n.Cn(t.j,r,D)),it=n.ys(h.skb((w.MAb(34,36,w.SAb(O).length),w.SAb(O).substr(34,2)))),it!=0&&n.tn(t.j,r,(n.Rk(),n.Pk)[n.Tl(t.j,r)]+it),y=n.ys(h.skb((w.MAb(36,39,w.SAb(O).length),w.SAb(O).substr(36,3)))),y!=0&&(y==4?n.wn(t.j,r,32):n.jn(t.j,r,4-y)),Qt=w.SAb(O).length<63?0:n.ys(h.skb((w.MAb(60,63,w.SAb(O).length),w.SAb(O).substr(60,3)))),n.rn(t.j,r,Qt,!1),v=w.SAb(O).length<45?0:n.ys(h.skb((w.MAb(42,45,w.SAb(O).length),w.SAb(O).substr(42,3)))),v){case 0:break;case 1:n.vn(t.j,r,768,!0);break;case 2:n.vn(t.j,r,128,!0);break;case 3:n.vn(t.j,r,384,!0);break;default:n.vn(t.j,r,896,!0);}w.SAb(O).length>=48&&(w.NAb(47,w.SAb(O).length),w.SAb(O).charCodeAt(47)==49)&&n.vn(t.j,r,L0,!0),c1=w.SAb(O).length<51?0:n.ys(h.skb((w.MAb(48,51,w.SAb(O).length),w.SAb(O).substr(48,3)))),c1!=0&&(D1==null&&(D1=B(o.Q7,H,6,ut,15,1)),D1[r]=c1);}for(E=0;E<Dt;E++){if((O=Rt.dhb(A))==null)return !1;P=h.Mib(h.skb((w.MAb(0,3,w.SAb(O).length),w.SAb(O).substr(0,3))),r1,qt)-1,i=h.Mib(h.skb((w.MAb(3,6,w.SAb(O).length),w.SAb(O).substr(3,3))),r1,qt)-1,l=h.Mib(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3))),r1,qt),Kt=w.SAb(O).length<12?0:n.ys(h.skb((w.MAb(9,12,w.SAb(O).length),w.SAb(O).substr(9,3)))),Bt=w.SAb(O).length<18?0:n.ys(h.skb((w.MAb(15,18,w.SAb(O).length),w.SAb(O).substr(15,3)))),l==8&&(t.k||n.Lm(t.j,P)||n.Lm(t.j,i))&&(l=9),n.ds(t,P,i,l,Kt,Bt);}for(S=0;S<ot;S++)if((O=Rt.dhb(A))==null)return !1;if((O=Rt.dhb(A))==null)return (t.i&1)!=0&&(t.f=n.lp(t.j)),n.js(t,D1),t.d||n.bw(t.j,15),!0;for(;O!=null&&!(h.akb(O,FA)||h.akb(O,le));){if(h.akb(w.SAb(O).substr(0,6),"M  CHG")&&(q=h.Mib(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3))),r1,qt),q>0))for(e=10,St=14,V=1;V<=q;++V,e+=8,St+=8)r=h.Mib(h.skb((w.MAb(e,e+3,w.SAb(O).length),w.SAb(O).substr(e,e+3-e))),r1,qt)-1,L=h.Mib(h.skb((w.MAb(St,St+3,w.SAb(O).length),w.SAb(O).substr(St,St+3-St))),r1,qt),n.jn(t.j,r,L);if(h.akb(w.SAb(O).substr(0,6),"M  ISO")&&(q=h.Mib(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3))),r1,qt),q>0))for(e=10,St=14,V=1;V<=q;++V,e+=8,St+=8)r=h.Mib(h.skb((w.MAb(e,e+3,w.SAb(O).length),w.SAb(O).substr(e,e+3-e))),r1,qt)-1,z=h.Mib(h.skb((w.MAb(St,St+3,w.SAb(O).length),w.SAb(O).substr(St,St+3-St))),r1,qt),n.tn(t.j,r,z);if(h.akb(w.SAb(O).substr(0,6),"M  RAD")&&(q=h.Mib(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3))),r1,qt),q>0))for(e=10,St=14,V=1;V<=q;++V,e+=8,St+=8)switch(r=h.Mib(h.skb((w.MAb(e,e+3,w.SAb(O).length),w.SAb(O).substr(e,e+3-e))),r1,qt)-1,wt=h.Mib(h.skb((w.MAb(St,St+3,w.SAb(O).length),w.SAb(O).substr(St,St+3-St))),r1,qt),wt){case 1:n.wn(t.j,r,16);break;case 2:n.wn(t.j,r,32);break;case 3:n.wn(t.j,r,48);}if((h.akb(w.SAb(O).substr(0,6),"M  RBC")||h.akb(w.SAb(O).substr(0,6),"M  RBD"))&&(q=h.Mib(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3))),r1,qt),q>0))for(e=10,St=14,V=1;V<=q;++V,e+=8,St+=8)switch(r=h.Mib(h.skb((w.MAb(e,e+3,w.SAb(O).length),w.SAb(O).substr(e,e+3-e))),r1,qt)-1,It=h.Mib(h.skb((w.MAb(St,St+3,w.SAb(O).length),w.SAb(O).substr(St,St+3-St))),r1,qt),It){case 3:case -1:n.vn(t.j,r,112,!0);break;case 1:n.vn(t.j,r,8,!0);break;case 2:n.vn(t.j,r,104,!0);break;case 4:n.vn(t.j,r,56,!0);}if(h.akb(w.SAb(O).substr(0,6),"M  ALS")&&(r=h.Mib(h.skb((w.MAb(7,10,w.SAb(O).length),w.SAb(O).substr(7,3))),r1,qt)-1,r>=0)){for(bt=h.Mib(h.skb((w.MAb(10,13,w.SAb(O).length),w.SAb(O).substr(10,3))),r1,qt),u=(w.NAb(14,w.SAb(O).length),w.SAb(O).charCodeAt(14)==84),c1=B(o.Q7,H,6,bt,15,1),e=16,V=0;V<bt;++V,e+=4)kt=h.skb((w.MAb(e,e+4,w.SAb(O).length),w.SAb(O).substr(e,e+4-e))),c1[V]=n.po(kt,1);n.Cn(t.j,r,6),n.qn(t.j,r,c1,u);}if(h.akb(w.SAb(O).substr(0,6),"M  SUB")&&(q=h.Mib(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3))),r1,qt),q>0)){for(e=10,St=14,V=1;V<=q;++V,e+=8,St+=8)if(r=h.Mib(h.skb((w.MAb(e,e+3,w.SAb(O).length),w.SAb(O).substr(e,e+3-e))),r1,qt)-1,xt=h.Mib(h.skb((w.MAb(St,St+3,w.SAb(O).length),w.SAb(O).substr(St,St+3-St))),r1,qt),xt==-2)n.vn(t.j,r,B1,!0);else if(xt>0){for(Nt=0,Q=0;Q<t.j.r;Q++)(n.Zl(t.j,0,Q)==r||n.Zl(t.j,1,Q)==r)&&++Nt;xt>Nt&&n.vn(t.j,r,Z1,!0);}}if(h.akb(w.SAb(O).substr(0,6),"M  RGP")&&(q=h.Mib(h.skb((w.MAb(6,9,w.SAb(O).length),w.SAb(O).substr(6,3))),r1,qt),q>0))for(e=10,St=14,V=1;V<=q;++V,e+=8,St+=8)r=h.Mib(h.skb((w.MAb(e,e+3,w.SAb(O).length),w.SAb(O).substr(e,e+3-e))),r1,qt)-1,Et=h.Mib(h.skb((w.MAb(St,St+3,w.SAb(O).length),w.SAb(O).substr(St,St+3-St))),r1,qt),Et>=1&&Et<=20&&n.Cn(t.j,r,n.po("R"+Et,2));(h.akb(w.SAb(O).substr(0,6),"M  SAL")&&w.SAb(O).length>=17||h.akb(w.SAb(O).substr(0,6),"M  SDT")&&w.SAb(O).length>=12||h.akb(w.SAb(O).substr(0,6),"M  SED")&&w.SAb(O).length>=12)&&(!b&&(b=new U.Xwb()),Ht=(w.MAb(7,10,w.SAb(O).length),w.SAb(O).substr(7,3)),T=(G=Ht,U.Xnb(U.Kwb(b,G))),T||(T=new n.Es(),U.Pwb(b,Ht,T)),h.akb(w.SAb(O).substr(0,6),"M  SAL")&&h.akb(w.SAb(O).substr(10,w.SAb("  1").length),"  1")?T.a=h.Mib(h.skb((w.MAb(13,17,w.SAb(O).length),w.SAb(O).substr(13,4))),r1,qt):h.akb(w.SAb(O).substr(0,6),"M  SDT")?T.b=h.pkb((w.NAb(11,w.SAb(O).length+1),w.SAb(O).substr(11)),ir):h.akb(w.SAb(O).substr(0,6),"M  SED")&&(T.c=h.skb((w.NAb(11,w.SAb(O).length+1),w.SAb(O).substr(11))))),O=Rt.dhb(A);}}catch(L1){if(L1=C1(L1),Zt(L1,19))return g=L1,h.MB(g,(h.Pkb(),h.Okb),""),false;throw st(L1);}if(b)for(c=(Ut=new U.Epb(b).a.cd().Tb(),new U.Lpb(Ut));c.a.ed();)T=(k=c.a.fd(),k.jd()),T.b&&T.a!=0&&T.c!=null&&n.mn(t.j,T.a-1,T.c);return t.e&&(n.vs(t),n.es(t)),(t.i&1)!=0&&(t.f=n.lp(t.j)),n.js(t,D1),n.bw(t.j,15),true;},n.Bs=function(t,A){var e,r,P,i,D;for(t.b&&U.Iwb(t.b),t.c&&U.Iwb(t.c),P=0,r=Rt.dhb(A);r!=null&&h.akb(w.SAb(r).substr(0,7),Le);){for(r=h.skb((w.NAb(7,w.SAb(r).length+1),w.SAb(r).substr(7)));D=w.SAb("-").length,h.akb(w.SAb(r).substr(w.SAb(r).length-D,D),"-");){if(e=Rt.dhb(A),!h.akb(w.SAb(e).substr(0,7),Le))return  false;r=h.skb(h.Yjb(h.rkb(r,0,w.SAb(r).length-1),(w.NAb(7,w.SAb(e).length+1),w.SAb(e).substr(7))));}if(h.akb(w.SAb(r).substr(0,5),"BEGIN")){if(i=h.skb((w.NAb(6,w.SAb(r).length+1),w.SAb(r).substr(6))),h.akb(w.SAb(i).substr(0,4),"CTAB"))P=1;else if(h.akb(w.SAb(i).substr(0,4),"ATOM"))P=2;else if(h.akb(w.SAb(i).substr(0,4),"BOND"))P=3;else if(h.akb(w.SAb(i).substr(0,6),"SGROUP"))P=4;else if(h.akb(w.SAb(i).substr(0,10),"COLLECTION"))P=5;else return  false;}else if(h.akb(w.SAb(r).substr(0,3),"END"))P=0;else if(P==1)n.ss(t,r);else if(P==2)n.os(t,r);else if(P==3)n.qs(t,r);else if(P==4)n.us(t,r);else if(P==5)n.rs(t,r);else return  false;r=Rt.dhb(A);}for(;r!=null&&!(h.akb(w.SAb(r).substr(0,6),FA)||h.akb(r,le));)r=Rt.dhb(A);return  true;},n.Cs=function(){this.i=0;},n.Ds=function(){this.i=1;},x(127,1,{},n.Cs,n.Ds),s.a=false,s.d=false,s.e=false,s.g=false,s.i=0,s.k=false,o.z8=N(127),n.Es=function(){},x(224,1,{224:1},n.Es),s.a=0,s.b=false,o.y8=N(224),n.Fs=function(t){},n.Gs=function(t,A){var e,r;for(r=o0.Dnb(t.a,A),e=w.SAb(r).length;e<10;e++)h.Ekb(t.b,32);h.Ikb(t.b,r);},n.Hs=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It;for(tt=(h.Pkb(),d1),h.Ikb(t.b,"M  V30 BEGIN CTAB"+tt),h.Ikb(t.b,Z8+A.q+" "+A.r+" 0 0 0"+tt),h.Ikb(t.b,"M  V30 BEGIN ATOM"+tt),Q=0;Q<A.q;Q++){if(h.Ikb(t.b,Le+(Q+1)),(A.v==null?null:A.v[Q])!=null){for(T=A.v==null?null:A.v[Q],Qt=pt(J(A.B[Q],1),0),h.Ikb(t.b,Qt?" NOT[":" ["),V=0;V<T.length;V++)switch(V>0&&h.Ikb(t.b,","),G=(n.Rk(),n.Lk)[T[V]],w.SAb(G).length){case 2:case 3:case 1:h.Ikb(t.b,G);break;default:h.Ikb(t.b,"?");}h.Ikb(t.b,"]");}else pt(J(A.B[Q],1),0)?h.Ikb(t.b," A"):A.C[Q]>=129&&A.C[Q]<=144||A.C[Q]==154?h.Ikb(t.b," R#"):h.Ikb(t.b," "+(n.Rk(),n.Lk)[A.C[Q]]);if(e?(h.Ikb(t.b," "+Mt(Z0*t.c*A.J[Q].a)/Z0),h.Ikb(t.b," "+Mt(Z0*t.c*-A.J[Q].b)/Z0),h.Ikb(t.b," "+Mt(Z0*t.c*-A.J[Q].c)/Z0)):h.Ikb(t.b," 0 0 0"),h.Ikb(t.b," "+d.Math.abs(A.w[Q])),A.s[Q]!=0&&h.Ikb(t.b," CHG="+A.s[Q]),(A.u[Q]&48)!=0)switch(h.Ikb(t.b," RAD="),A.u[Q]&48){case 16:h.Ikb(t.b,"1");break;case 32:h.Ikb(t.b,"2");break;case 48:h.Ikb(t.b,"3");}((A.u[Q]&3)==1||(A.u[Q]&3)==2)&&(h.Ikb(t.b," CFG="),(A.u[Q]&3)==1?h.Ikb(t.b,"1"):h.Ikb(t.b,"2")),A.A[Q]!=0&&h.Ikb(t.b," MASS="+A.A[Q]),It=((A.u[Q]&bA)>>>27)-1,It!=-1&&h.Ikb(t.b," VAL="+(It==0?"-1":h.qjb(It))),c=A.C[Q],(c>=129&&c<=144||c==154)&&h.Ikb(t.b," RGROUPS=(1 "+(c==154?0:c>=142?c-141:c-125)+")"),q=J(X0,A.B[Q]),nt(q,384)==0?h.Ikb(t.b," HCOUNT=2"):nt(q,128)==0?h.Ikb(t.b," HCOUNT=1"):nt(q,1792)==0?h.Ikb(t.b," HCOUNT=-1"):nt(q,1664)==0&&h.Ikb(t.b," HCOUNT=1"),Ut=J(A.B[Q],6144),nt(Ut,0)!=0&&(pt(J(Ut,Z1),0)?h.Ikb(t.b," SUBST="+(A.e[Q]+1)):h.Ikb(t.b," SUBST=-1")),Dt=J(A.B[Q],120),nt(Dt,0)!=0&&(nt(Dt,112)==0?h.Ikb(t.b," RBCNT=-1"):nt(Dt,8)==0||nt(Dt,104)==0?h.Ikb(t.b," RBCNT=2"):nt(Dt,88)==0?h.Ikb(t.b," RBCNT=3"):nt(Dt,56)==0&&h.Ikb(t.b," RBCNT=4")),h.Ikb(t.b,tt);}for(h.Ikb(t.b,"M  V30 END ATOM"+tt),h.Ikb(t.b,"M  V30 BEGIN BOND"+tt),g=0;g<A.r;g++){switch(h.Ikb(t.b,Le+(g+1)),A.H[g]){case 1:lt=1,bt=0;break;case 2:lt=2,bt=0;break;case 4:lt=3,bt=0;break;case 129:lt=1,bt=3;break;case 257:lt=1,bt=1;break;case 386:lt=2,bt=2;break;case 8:lt=4,bt=0;break;case 16:lt=9,bt=0;break;default:lt=1,bt=0;}v=A.G[g]&31,v!=0&&(v==8?lt=4:v==3?lt=5:v==9?lt=6:v==10?lt=7:lt=8),h.Ikb(t.b," "+lt+" "+(A.D[0][g]+1)+" "+(A.D[1][g]+1)),bt!=0&&h.Ikb(t.b," CFG="+bt),ot=A.G[g]&384,wt=ot==0?0:ot==256?1:2,wt!=0&&h.Ikb(t.b," TOPO="+wt),h.Ikb(t.b,tt);}for(h.Ikb(t.b,"M  V30 END BOND"+tt),p=false,l=0;l<A.q;l++)if((A.t==null||A.t[l]==null?null:h.$jb((K=A.t[l],q1.Cnb(),K)))!=null){p=true;break;}if(p){for(h.Ikb(t.b,"M  V30 BEGIN SGROUP"+tt),S=0,L=0;L<A.q;L++)(A.t==null||A.t[L]==null?null:h.$jb((O=A.t[L],q1.Cnb(),O)))!=null&&(++S,h.Ikb(t.b,Le+S+" DAT "+S+" ATOMS=(1 "+(L+1)+') FIELDNAME="'+ir+'" -'+tt),h.Ikb(t.b,'M  V30 FIELDDISP="'),n.Gs(t,e?t.c*A.J[L].a:0),n.Gs(t,e?t.c*A.J[L].b:0),h.Ikb(t.b,'    DA    ALL  1       5" FIELDDATA="'+(A.t==null||A.t[L]==null?null:h.$jb((K=A.t[L],q1.Cnb(),K)))+'"'+tt));h.Ikb(t.b,"M  V30 END SGROUP"+tt);}for(ut=false,r=0,z=B(o.Q7,H,6,32,15,1),i=B(o.Q7,H,6,32,15,1),y=0;y<A.f;y++)((A.u[y]&3)==1||(A.u[y]&3)==2)&&(ut=true,ct=(A.u[y]&l1)>>18,ct==1?++i[(A.u[y]&l1)>>18!=1&&(A.u[y]&l1)>>18!=2?-1:(A.u[y]&F0)>>20]:ct==2?++z[(A.u[y]&l1)>>18!=1&&(A.u[y]&l1)>>18!=2?-1:(A.u[y]&F0)>>20]:++r);for(P=0,it=B(o.Q7,H,6,32,15,1),D=B(o.Q7,H,6,32,15,1),k=0;k<A.g;k++)n.fm(A,k)!=2&&((A.F[k]&3)==1||(A.F[k]&3)==2)&&(ut=true,ct=(A.F[k]&768)>>8,ct==1?++D[(A.F[k]&768)>>8!=1&&(A.F[k]&768)>>8!=2?-1:(A.F[k]&E0)>>10]:ct==2?++it[(A.F[k]&768)>>8!=1&&(A.F[k]&768)>>8!=2?-1:(A.F[k]&E0)>>10]:++P);if(ut){if(h.Ikb(t.b,"M  V30 BEGIN COLLECTION"+tt),r!=0){for(h.Ikb(t.b,"M  V30 MDLV30/STEABS ATOMS=("+r),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&l1)>>18==0&&h.Ikb(t.b," "+(u+1));h.Ikb(t.b,")"+tt);}if(P!=0){for(h.Ikb(t.b,"M  V30 MDLV30/STEABS BONDS=("+P),b=0;b<A.g;b++)n.fm(A,b)!=2&&((A.F[b]&3)==1||(A.F[b]&3)==2)&&(A.F[b]&768)>>8==0&&h.Ikb(t.b," "+(b+1));h.Ikb(t.b,")"+tt);}for(E=0;E<32;E++){if(z[E]!=0){for(h.Ikb(t.b,Y8+(E+1)+" ATOMS=("+z[E]),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&l1)>>18==2&&((A.u[u]&l1)>>18!=1&&(A.u[u]&l1)>>18!=2?-1:(A.u[u]&F0)>>20)==E&&h.Ikb(t.b," "+(u+1));h.Ikb(t.b,")"+tt);}if(i[E]!=0){for(h.Ikb(t.b,X8+(E+1)+" ATOMS=("+i[E]),u=0;u<A.f;u++)((A.u[u]&3)==1||(A.u[u]&3)==2)&&(A.u[u]&l1)>>18==1&&((A.u[u]&l1)>>18!=1&&(A.u[u]&l1)>>18!=2?-1:(A.u[u]&F0)>>20)==E&&h.Ikb(t.b," "+(u+1));h.Ikb(t.b,")"+tt);}if(it[E]!=0){for(h.Ikb(t.b,Y8+(E+1)+" BONDS=("+it[E]),b=0;b<A.g;b++)n.fm(A,b)!=2&&((A.F[b]&3)==1||(A.F[b]&3)==2)&&(A.F[b]&768)>>8==2&&((A.F[b]&768)>>8!=1&&(A.F[b]&768)>>8!=2?-1:(A.F[b]&E0)>>10)==E&&h.Ikb(t.b," "+(b+1));h.Ikb(t.b,")"+tt);}if(D[E]!=0){for(h.Ikb(t.b,X8+(E+1)+" BONDS=("+D[E]),b=0;b<A.g;b++)n.fm(A,b)!=2&&((A.F[b]&3)==1||(A.F[b]&3)==2)&&(A.F[b]&768)>>8==1&&((A.F[b]&768)>>8!=1&&(A.F[b]&768)>>8!=2?-1:(A.F[b]&E0)>>10)==E&&h.Ikb(t.b," "+(b+1));h.Ikb(t.b,")"+tt);}}h.Ikb(t.b,"M  V30 END COLLECTION"+tt);}h.Ikb(t.b,LP+tt);},n.Is=function(){n.Fs(this),this.b=new h.Lkb();},n.Js=function(t){n.Ks.call(this,t);},n.Ks=function(t){n.Ls.call(this,t,new h.Lkb());},n.Ls=function(t,A){var e,r,P,i;n.Fs(this),this.a=new o0.Inb((r=Pr,r)),n.bw(t,15),i=(h.Pkb(),d1),this.b=A,P=t.P!=null?t.P:"",h.Ikb(this.b,P+i),h.Ikb(this.b,"Actelion Java MolfileCreator 2.0"+i+i),h.Ikb(this.b,"  0  0  0  0  0  0              0 V3000"+i),this.c=1,e=n.Ns(t),e&&(this.c=n.Ms(t)),n.Hs(this,t,e),h.Ikb(this.b,FA+i);},n.Ms=function(t){var A,e,r,P,i,D,u,Q,l;if(l=1,r=n.Wl(t,t.q,t.r,(n.Rk(),n.Qk)),r!=0)(r<1||r>3)&&(l=1.5/r);else {for(Q=_8,A=1;A<t.q;A++)for(e=0;e<A;e++)i=t.J[e].a-t.J[A].a,D=t.J[e].b-t.J[A].b,u=t.J[e].c-t.J[A].c,P=i*i+D*D+u*u,Q>P&&(Q=P);l=3/d.Math.max(.75,Q);}return l;},n.Ns=function(t){var A;if(t.q==1)return  true;for(A=1;A<t.q;A++)if(t.J[A].a!=t.J[0].a||t.J[A].b!=t.J[0].b||t.J[A].c!=t.J[0].c)return  true;return  false;},n.Os=function(t,A){var e;return e=new n.Is(),e.c=A,n.bw(t,15),n.Hs(e,t,true),e.b.a;},x(150,1,{},n.Is,n.Js),s.c=1,o.A8=N(150),n.Rs=function(){n.Rs=f1,n.Ps=j(F(o.Ocb,2),b0,13,0,[j(F(o.Ocb,1),yt,2,6,["Acyl","gCaHA`AIf`@"]),j(F(o.Ocb,1),yt,2,6,["Alloc","gNph@l@ILzuR@@"]),j(F(o.Ocb,1),yt,2,6,["Allyl","Al"]),j(F(o.Ocb,1),yt,2,6,["Benzyl;Bn","daD@`F@DjUZxHH@@"]),j(F(o.Ocb,1),yt,2,6,["Boc","daxD`@S@AIgijj@@"]),j(F(o.Ocb,1),yt,2,6,["BOM;BzOM","deTH`@H@Re[TYj`@@@"]),j(F(o.Ocb,1),yt,2,6,["Bs","dmtDpAdLS`aPAIe]jf`@e`@@"]),j(F(o.Ocb,1),yt,2,6,["Bt","dew@`@aJ@DiY]paej`@@@"]),j(F(o.Ocb,1),yt,2,6,["Btm","did@P@BJ@Die_ahH@@@"]),j(F(o.Ocb,1),yt,2,6,["Butyl;nButyl","gJPHB@IRuP@"]),j(F(o.Ocb,1),yt,2,6,["Benzoyl;Bz","didH`@p@RYm^Eh@@@@"]),j(F(o.Ocb,1),yt,2,6,["Bzh","dg|@`N@LdbRbtJUB]aAP@@@@"]),j(F(o.Ocb,1),yt,2,6,["Cbz","dmtD`@S@AIgYVUZh@@@@"]),j(F(o.Ocb,1),yt,2,6,["cButyl","gKPHL@IThuT@@"]),j(F(o.Ocb,1),yt,2,6,["cHeptyl","daD@`L@DjWVzjj`@"]),j(F(o.Ocb,1),yt,2,6,["cHexyl","gOpHL@IToWUU@@"]),j(F(o.Ocb,1),yt,2,6,["cOctyl","did@`L@DjWWajjj@@"]),j(F(o.Ocb,1),yt,2,6,["cPentyl","gFpHL@ITimUP@"]),j(F(o.Ocb,1),yt,2,6,["cPropyl","gBPHL@Qxjh@"]),j(F(o.Ocb,1),yt,2,6,["COOH","gC`h@l@ILt@@ !Bb@K~@Hc}"]),j(F(o.Ocb,1),yt,2,6,["DEAE","daz@`@x@RiUjj`@"]),j(F(o.Ocb,1),yt,2,6,["DEIPS","diD@P@\\B@DjfVjj`@"]),j(F(o.Ocb,1),yt,2,6,["DMIPS","gNpD@xD@RjZjh@"]),j(F(o.Ocb,1),yt,2,6,["DMPM","dcLD`@kPCIEMDdcttDDT@@"]),j(F(o.Ocb,1),yt,2,6,["DMPS","deT@P@\\B@LddTjPsU@@@@"]),j(F(o.Ocb,1),yt,2,6,["DMTr","fak@b@@Mt@ISZ{SMjo{NQKfm@AU@@@E@@@@"]),j(F(o.Ocb,1),yt,2,6,["DNP","dkmB`hdDt~@HeNfS{HihheCAUhBHX@@"]),j(F(o.Ocb,1),yt,2,6,["DNS;Dan","fhi`a@KPP@HH@YIHYheEhYKQgKP@@QP@@@"]),j(F(o.Ocb,1),yt,2,6,["DPIPS","fdyAA@H@\\B@FRRIQSQIHzp_Qjh@h@@@@@"]),j(F(o.Ocb,1),yt,2,6,["DPTBS","fleAA@H@\\B@FRRIQSRIIWNbEMU@EP@@@@@"]),j(F(o.Ocb,1),yt,2,6,["DTBMS","dmT@P@\\B@Djffjjjh@@"]),j(F(o.Ocb,1),yt,2,6,["Ethyl","eMBD@ch@"]),j(F(o.Ocb,1),yt,2,6,["Fmoc","fde@b@@Hp@IL{LrjxeVCzKUT@@@P@@@"]),j(F(o.Ocb,1),yt,2,6,["iAmyl","gGPHJ@YIDZj@@"]),j(F(o.Ocb,1),yt,2,6,["iButyl","gJPHB@ITuP@"]),j(F(o.Ocb,1),yt,2,6,["Im","gFtHAj@IRnKSP@"]),j(F(o.Ocb,1),yt,2,6,["iPropyl","gC`HL@Qz`@"]),j(F(o.Ocb,1),yt,2,6,["MDIPS","diD@P@\\B@DjfZjj`@"]),j(F(o.Ocb,1),yt,2,6,["MDPS","foA@A@@NA@CIIEEBdeeVLzj@@@@@@"]),j(F(o.Ocb,1),yt,2,6,["Methyl","eFBH@c@@"]),j(F(o.Ocb,1),yt,2,6,["MEM","gNphAR@IRoUT@@"]),j(F(o.Ocb,1),yt,2,6,["Mes","deT@`J@DjY{[`bB`@@"]),j(F(o.Ocb,1),yt,2,6,["MMTr","ffcAB@@Z@Dim]ifuWYrI\\uh@Jh@@@@@@"]),j(F(o.Ocb,1),yt,2,6,["MOM","gCaHA`AJZ`@"]),j(F(o.Ocb,1),yt,2,6,["MPM;PMB","deTH`@d@Rfuunh@J@@"]),j(F(o.Ocb,1),yt,2,6,["Ms","gJPdH`DD@cuh@"]),j(F(o.Ocb,1),yt,2,6,["MTM","gC`D@DX@Rfh@"]),j(F(o.Ocb,1),yt,2,6,["mTolyl","daD@`N@DjWjXHB@@"]),j(F(o.Ocb,1),yt,2,6,["N3","gClHaE`@RnReX@"]),j(F(o.Ocb,1),yt,2,6,["nAmyl;Amyl","gGPHJ@IRmU@@"]),j(F(o.Ocb,1),yt,2,6,["neoAm","gGPHJ@IUMU@@"]),j(F(o.Ocb,1),yt,2,6,["NO2,Nitro","gChhhE`BRnRYh@"]),j(F(o.Ocb,1),yt,2,6,["Np","deVDaHAI@HeNR[e_aZ@B@@"]),j(F(o.Ocb,1),yt,2,6,["nPropyl;Propyl","gC`HL@IST@@"]),j(F(o.Ocb,1),yt,2,6,["oTolyl","daD@`J@DjYvxH`@@"]),j(F(o.Ocb,1),yt,2,6,["Phenyl","gOpHL@IToVD@@@"]),j(F(o.Ocb,1),yt,2,6,["Pht","dcLL`@RU@Dfyed]ZBA`@@"]),j(F(o.Ocb,1),yt,2,6,["Piv;Pv","gNqHA`AIffj`@"]),j(F(o.Ocb,1),yt,2,6,["PMBM","dcLD`@T`AJUm]FZh@J@@"]),j(F(o.Ocb,1),yt,2,6,["PNB","dcNLaHAEt@bTyInUvxV`@f@@"]),j(F(o.Ocb,1),yt,2,6,["Poc","didD`@S@AIgexVjj`@"]),j(F(o.Ocb,1),yt,2,6,["PPi","diDFsHSB[`|J|A@Lxn{lddqdZih@@"]),j(F(o.Ocb,1),yt,2,6,["pTolyl","daD@`N@DjWzXHB@@"]),j(F(o.Ocb,1),yt,2,6,["sAmyl","gGPHL@YIDZj@@"]),j(F(o.Ocb,1),yt,2,6,["sButyl","gJPHL@ITuP@"]),j(F(o.Ocb,1),yt,2,6,["SEM","diDHPFApD@rRQUJjj`@"]),j(F(o.Ocb,1),yt,2,6,["SES","dedDpHP@``AgCIICeHmUT@@"]),j(F(o.Ocb,1),yt,2,6,["SO3H","gJQdHl@``D^m@@"]),j(F(o.Ocb,1),yt,2,6,["tAmyl","gGPHB@IUMU@@"]),j(F(o.Ocb,1),yt,2,6,["TBDMS;TBS","dax@P@\\B@Djfjjh@@"]),j(F(o.Ocb,1),yt,2,6,["TBDPS","fdy@A@@NA@CIIEEEIde]XOhuPAT@@@@@"]),j(F(o.Ocb,1),yt,2,6,["TBMPS","dg\\HPHApH@rRQJJPjg]UAT@@@"]),j(F(o.Ocb,1),yt,2,6,["tButyl,tBu","gJPHB@Q}T@@"]),j(F(o.Ocb,1),yt,2,6,["TDS","ded@P@\\B@LddTeeUUP@@"]),j(F(o.Ocb,1),yt,2,6,["Tf","daxDhHP@``BiAiCiCIICHmU@@"]),j(F(o.Ocb,1),yt,2,6,["TFA","gNqBJIARFdF@YEHYUL@@"]),j(F(o.Ocb,1),yt,2,6,["Thexyl","gNpHB@IUMUT@@"]),j(F(o.Ocb,1),yt,2,6,["THF","gFqH@PAJYujj@@"]),j(F(o.Ocb,1),yt,2,6,["THP","gOqH@PAJYZzjh@"]),j(F(o.Ocb,1),yt,2,6,["TIPS","dmT@P@\\B@DjfYjjjh@@"]),j(F(o.Ocb,1),yt,2,6,["TMS","gJPD@xD@czh@"]),j(F(o.Ocb,1),yt,2,6,["Tosyl;Ts","dmtDPHP@``CIICLeaeZ@B@@"]),j(F(o.Ocb,1),yt,2,6,["Troc","diDDHJxHaHcH`PCHiBeJjf@@"]),j(F(o.Ocb,1),yt,2,6,["Trt","fbm@B@A@FRQIRKQPiIZdoIcdHJ`@@@@@@"]),j(F(o.Ocb,1),yt,2,6,["Xyl","did@`J@DjYynBHH@@"])]);},n.Ss=function(){n.Rs();var t,A,e,r,P,i,D,u,Q;for(n.Qs=new U.Xwb(),D=n.Ps,u=0,Q=D.length;u<Q;++u)for(i=D[u],P=h.okb(i[0],";"),A=P,e=0,r=A.length;e<r;++e)t=A[e],U.Pwb(n.Qs,h.mkb(w.SAb(t).toLowerCase(),"-"),i[1]);},n.Ts=function(t){n.Rs();var A,e,r,P,i,D,u,Q,l,L;if(t==null||w.SAb(t).length==0)return null;for(D=null,t=h.mkb(w.SAb(t).toLowerCase(),"-"),Q=n.Ps,l=0,L=Q.length;l<L;++l)for(u=Q[l],i=h.okb(u[0],";"),e=i,r=0,P=e.length;r<P;++r){if(A=e[r],h.akb(h.mkb(w.SAb(A).toLowerCase(),"-"),t))return A;if(h.pkb(h.mkb(w.SAb(A).toLowerCase(),"-"),t))if(D==null)D=A;else return "";}return D;},n.Qs=null,n.Ws=function(){var t,A,e,r;for(this.b=new U.pub(),this.c=new U.pub(),this.d=new U.pub(),n.Us=j(F(o.m8,1),R1,18,0,[new n.mk(1,"Hydrogen","H",2.2),new n.mk(2,"Helium","He",0),new n.mk(3,"Lithium","Li",.98),new n.mk(4,"Beryllium","Be",1.57),new n.mk(5,"Boron","B",2.04),new n.mk(6,"Carbon","C",2.55),new n.mk(7,"Nitrogen","N",3.04),new n.mk(8,"Oxygen","O",3.44),new n.mk(9,"Fluorine","F",3.98),new n.mk(10,"Neon","Ne",0),new n.mk(11,"Sodium","Na",.93),new n.mk(12,"Magnesium","Mg",1.31),new n.mk(13,"Aluminum","Al",1.61),new n.mk(14,"Silicon","Si",1.9),new n.mk(15,"Phosphorus","P",2.19),new n.mk(16,"Sulfur","S",2.58),new n.mk(17,"Chlorine","Cl",3.16),new n.mk(18,"Argon","Ar",0),new n.mk(19,"Potassium","K",1),new n.mk(20,"Calcium","Ca",1.36),new n.mk(21,"Scandium","Sc",1.54),new n.mk(22,"Titanium","Ti",1.63),new n.mk(23,"Vanadium","V",1.66),new n.mk(24,"Chromium","Cr",1.55),new n.mk(25,"Manganese","Mn",1.83),new n.mk(26,"Iron","Fe",1.88),new n.mk(27,"Cobalt","Co",1.91),new n.mk(28,"Nickel","Ni",1.9),new n.mk(29,"Copper","Cu",1.65),new n.mk(30,"Zinc","Zn",1.81),new n.mk(31,"Gallium","Ga",2.01),new n.mk(32,"Germanium","Ge",0),new n.mk(33,"Arsenic","As",2.18),new n.mk(34,"Selenium","Se",2.55),new n.mk(35,"Bromine","Br",0),new n.mk(36,"Krypton","Kr",2.96),new n.mk(37,"Rubidium","Rb",0),new n.mk(38,"Strontium","Sr",.82),new n.mk(39,"Yttrium","Y",.95),new n.mk(40,"Zirconium","Zr",1.22),new n.mk(41,"Niobium","Nb",1.33),new n.mk(42,"Molybdenum","Mo",1.6),new n.mk(43,"Technetium","Tc",2.16),new n.mk(44,"Ruthenium","Ru",1.9),new n.mk(45,"Rhodium","Rh",2.2),new n.mk(46,"Palladium","Pd",2.28),new n.mk(47,"Silver","Ag",2.2),new n.mk(48,"Cadmium","Cd",1.93),new n.mk(49,"Indium","In",1.69),new n.mk(50,"Tin","Sn",1.78),new n.mk(51,"Antimony","Sb",1.96),new n.mk(52,"Tellurium","Te",0),new n.mk(53,"Iodine","I",2.05),new n.mk(54,"Xenon","Xe",2.1),new n.mk(55,"Cesium","Cs",2.66),new n.mk(56,"Barium","Ba",2.6),new n.mk(57,"Lanthanum","La",.79),new n.mk(58,"Cerium","Ce",.89),new n.mk(59,"Praesodymium","Pr",1.1),new n.mk(60,"Neodymium","Nd",1.12),new n.mk(61,"Promethium","Pm",1.13),new n.mk(62,"Samarium","Sm",1.14),new n.mk(63,"Europium","Eu",0),new n.mk(64,"Gadolinium","Gd",1.17),new n.mk(65,"Terbium","Tb",0),new n.mk(66,"Dyprosium","Dy",1.2),new n.mk(67,"Holmium","Ho",0),new n.mk(68,"Erbium","Er",1.22),new n.mk(69,"Thulium","Tm",1.23),new n.mk(70,"Ytterbium","Yb",1.24),new n.mk(71,"Lutetium","Lu",1.25),new n.mk(72,"Hafnium","Hf",0),new n.mk(73,"Tantalium","Ta",1.27),new n.mk(74,"Wolfram","W",1.3),new n.mk(75,"Rhenium","Re",1.5),new n.mk(76,"Osmium","Os",2.36),new n.mk(77,"Iridium","Ir",1.9),new n.mk(78,"Platinum","Pt",2.2),new n.mk(79,"Gold","Au",2.2),new n.mk(80,"Mercury","Hg",2.28),new n.mk(81,"Thallium","Tl",2.54),new n.mk(82,"Lead","Pb",2),new n.mk(83,"Bismuth","Bi",1.62),new n.mk(84,"Polonium","Po",2.33),new n.mk(85,"Astatine","At",2.02),new n.mk(86,"Radon","Rn",2),new n.mk(87,"Francium","Fr",2.2),new n.mk(88,"Radium","Ra",0),new n.mk(89,"Actinium","Ac",.7),new n.mk(90,"Thorium","Th",.89),new n.mk(91,"Protactinium","Pa",1.1),new n.mk(92,"Uranium","U",1.3),new n.mk(93,"Neptunium","Np",1.5),new n.mk(94,"Plutonium","Pu",1.38),new n.mk(95,"Americium","Am",1.36),new n.mk(96,"Curium","Cm",1.28),new n.mk(97,"Berkelium","Bk",1.3),new n.mk(98,"Californium","Cf",1.3),new n.mk(99,"Einsteinium","E",1.3),new n.mk(100,"Fermium","Fm",1.3),new n.mk(101,"Mendelevium","Mv",1.3)]),A=n.Us,e=0,r=A.length;e<r;++e)t=A[e],U.qob(this.b,h.qjb(t.c),t),U.rob(this.c,t.b,t),U.rob(this.d,w.SAb(t.d).toLowerCase(),t);this.a=B(o.vfb,Tt,6,n.Us.length,16,1),this.a[3]=true,this.a[11]=true,this.a[19]=true,this.a[37]=true,this.a[55]=true,this.a[87]=true;},n.Xs=function(t){return U.nob((!n.Vs&&(n.Vs=new n.Ws()),n.Vs).b,h.qjb(t));},x(401,1,{},n.Ws),s.pb=function(){return "";},o.B8=N(401),n.Zs=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g;for(l=t.i.M,L=0,i=0;i<e;i++)l>A[i]&&(l=A[i],L=i);for(g=B(o.Q7,H,6,e,15,1),Q=L>0?L-1:e-1,y=L<e-1?L+1:0,D=A[Q]<A[y],P=0;P<e;P++)g[P]=A[L],D?--L<0&&(L=e-1):++L==e&&(L=0);for(b=new U.rrb(t.j);b.a<b.c.a.length;)if(c=U.qrb(b),c.length==e){for(r=true,u=0;u<e;u++)if(c[u]!=g[u]){r=false;break;}if(r)return;}U.sj(t.j,g),T=n.jt(t,g),U.sj(t.k,T),n.Bt(t,g,T);},n.$s=function(t,A,e){var r,P,i,D,u,Q,l;for(Q=B(o.Q7,H,6,t.g,15,1),D=B(o.Q7,H,6,t.g,15,1),l=B(o.vfb,Tt,6,t.i.f,16,1),r=n.Zl(t.i,0,A),P=n.Zl(t.i,1,A),Q[0]=r,Q[1]=P,D[1]=-1,l[P]=true,u=1;u>=1;){if(++D[u],D[u]==n.$o(t.i,Q[u])){l[Q[u]]=false,--u;continue;}if(i=n.Zo(t.i,Q[u],D[u]),!(l[i]||e[i])){if(i==r&&u>1){if(n.Zs(t,Q,u+1),t.j.a.length>=f0)return;continue;}u+1<t.g&&(++u,Q[u]=i,l[i]=true,D[u]=-1);}}},n._s=function(t,A,e,r,P,i,D,u){var Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct;for(lt=U.xj(t.j,A),y=lt,T=0,c=y.length;T<c;++T)if(L=y[T],!n.xt(t,L))return  true;for(ut=U.xj(t.k,A),Dt=ut.length,g=0,l=0,ct=false,G=0;G<Dt;G++)g<<=1,l<<=1,n.zt(t,ut[G])?g|=1:u&&n.wt(t,ut[G])?(g|=1,l|=1):(Q=e[A][G],Q!=-1&&(r[Q]?P[Q]&&(g|=1,i[Q]||(l|=1)):ct=true));switch(q=false,Dt){case 3:for(k=j(F(o.Q7,1),H,6,15,[2,1,4]),q=true,E=0;E<3;E++)(g&k[E])==k[E]&&(n.Tl(t.i,lt[E])==6&&n.Bl(t.i,lt[E])==1||n.Tl(t.i,lt[E])==5&&n.Bl(t.i,lt[E])==0)&&(P[A]=true,D[A]=E,(l&k[E])==0&&(q=false));break;case 5:for(v=j(F(o.Q7,1),H,6,15,[10,5,18,9,20]),q=true,it=0;it<5;it++)if((g&v[it])==v[it])switch(n.Tl(t.i,lt[it])){case 6:n.Bl(t.i,lt[it])==-1&&(P[A]=true,D[A]=it,(l&v[it])==0&&(q=false));break;case 7:n.Bl(t.i,lt[it])<=0&&(P[A]=true,D[A]=it);break;case 8:P[A]=true,D[A]=it;break;case 16:case 34:case 52:n.$o(t.i,lt[it])==2&&(P[A]=true,D[A]=it);}break;case 6:q=true,(g&21)==21&&(P[A]=true,(l&21)==0&&(q=false)),(g&42)==42&&(P[A]=true,(l&42)==0&&(q=false));break;case 7:for(S=j(F(o.Q7,1),H,6,15,[42,21,74,37,82,41,84]),q=true,p=0;p<7;p++)if((g&S[p])==S[p]){if(n.Tl(t.i,lt[p])==6&&(n.Bl(t.i,lt[p])==1||u&&n.mt(t,lt[p]))||n.Tl(t.i,lt[p])==5&&n.Bl(t.i,lt[p])==0)P[A]=true,D[A]=p,(l&S[p])==0&&(q=false);else for(V=0;V<2;V++)if(bt=(p+6+V)%7,ot=ut[bt],Ut=e[A][bt],Ut!=-1&&!P[Ut]&&U.xj(t.k,Ut).length==5){for(wt=U.xj(t.k,Ut),K=0;wt[K]!=ot;)++K;for(O=false,z=0,tt=1;tt<=4;tt++){if(b=wt[(K+tt)%5],Qt=n.fm(t.i,b)>1||n.im(t.i,b)==8||u&&n.wt(t,b),Qt&&O){z=0;break;}O=Qt,O&&++z;}if(z==2){P[A]=true,P[Ut]=true,i[A]=true,i[Ut]=true,U.sj(t.b,h.qjb(ot));break;}}}}return P[A]&&!q&&(i[A]=true),P[A]?true:!ct;},n.at=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b,g,k;for(i=B(o.Q7,E1,7,t.j.a.length,0,2),l=0;l<t.j.a.length;l++)for(i[l]=B(o.Q7,H,6,U.xj(t.j,l).length,15,1),L=0;L<U.xj(t.j,l).length;L++)i[l][L]=-1;for(g=B(o.Q7,H,6,t.i.g,15,1),c=0;c<t.k.a.length;c++)if(b=U.xj(t.k,c),b.length==3||b.length>=5&&b.length<=7)for(Q=0;Q<b.length;Q++)u=b[Q],n.$o(t.i,n.Zl(t.i,0,u))==3&&n.$o(t.i,n.Zl(t.i,1,u))==3&&(g[u]>0?(i[g[u]>>>16][g[u]&32767]=c,i[c][Q]=g[u]>>>16):g[u]=(c<<16)+LA+Q);for(D=B(o.vfb,Tt,6,t.j.a.length,16,1),k=0,y=-1;k>y;)for(y=k,T=0;T<t.j.a.length;T++)D[T]||n._s(t,T,i,D,A,e,r,P)&&(D[T]=true,++k);},n.bt=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;for(P=n.Zl(t.i,0,A),i=n.Zl(t.i,1,A),Q=B(o.Q7,H,6,t.i.f,15,1),l=B(o.Q7,H,6,t.i.f,15,1),L=B(o.Q7,H,6,t.i.f,15,1),Q[0]=P,Q[1]=i,l[P]=1,l[i]=2,L[P]=-1,L[i]=P,u=1,y=1;u<=y;){for(T=0;T<n.$o(t.i,Q[u]);T++){if(D=n.Zo(t.i,Q[u],T),u>1&&D==P){for(b=B(o.Q7,H,6,l[Q[u]],15,1),r=Q[u],c=0;c<b.length;c++)b[c]=r,r=L[r];return b;}l[D]==0&&!e[D]&&(Q[++y]=D,l[D]=l[Q[u]]+1,L[D]=Q[u]);}++u;}return null;},n.ct=function(t,A,e){var r,P;for(P=U.xj(t.j,A),r=0;r<P.length;r++)if(e==P[r])return r;return  -1;},n.dt=function(t,A){return t.a[A]&e1;},n.et=function(t,A,e){var r,P;for(P=U.xj(t.k,A),r=0;r<P.length;r++)if(e==P[r])return r;return  -1;},n.ft=function(t,A){return t.c[A]&e1;},n.gt=function(t,A){return t.d[A];},n.ht=function(t,A){return U.xj(t.j,A);},n.it=function(t,A){return U.xj(t.k,A);},n.jt=function(t,A){var e,r,P,i,D;for(i=A.length,D=B(o.Q7,H,6,i,15,1),r=0;r<i;r++)for(e=r==i-1?A[0]:A[r+1],P=0;P<n.$o(t.i,A[r]);P++)if(n.Zo(t.i,A[r],P)==e){D[r]=n._o(t.i,A[r],P);break;}return D;},n.kt=function(t,A){return U.xj(t.k,A).length;},n.lt=function(t,A,e){var r;for(r=0;r<t.k.a.length;r++)if(n.rt(t,r,A)&&n.rt(t,r,e))return r;return  -1;},n.mt=function(t,A){var e;for(e=0;e<n.$o(t.i,A);e++)if(n.ap(t.i,A,e)==2&&n.Tl(t.i,n.Zo(t.i,A,e))==8)return  true;return  false;},n.nt=function(t,A){return t.e[A];},n.ot=function(t,A){return (t.a[A]&H1)!=0;},n.pt=function(t,A){return (t.c[A]&H1)!=0;},n.qt=function(t,A,e){var r,P,i,D;for(P=U.xj(t.j,A),i=0,D=P.length;i<D;++i)if(r=P[i],e==r)return  true;return  false;},n.rt=function(t,A,e){var r,P,i,D;for(P=U.xj(t.k,A),i=0,D=P.length;i<D;++i)if(r=P[i],e==r)return  true;return  false;},n.st=function(t,A){return t.f[A];},n.tt=function(t,A){return (t.a[A]&r0)!=0;},n.ut=function(t,A){return (t.c[A]&r0)!=0;},n.vt=function(t,A){return (t.a[A]&yA)!=0;},n.wt=function(t,A){var e,r,P,i,D,u;for(D=0;D<2;D++)if(e=n.Zl(t.i,D,A),n.Tl(t.i,e)==7&&n.$o(t.i,e)==2&&(r=n.Zl(t.i,1-D,A),n.Tl(t.i,r)==6)){for(u=0;u<n.$o(t.i,r);u++)if(P=n.Zo(t.i,r,u),i=n._o(t.i,r,u),(n.Tl(t.i,P)==8||n.Tl(t.i,P)==16)&&n.fm(t.i,i)==2&&n.$o(t.i,P)==1)return  true;}return  false;},n.xt=function(t,A){var e,r,P,i,D;if(t.i.K){if(pt(J(n.Ol(t.i,A),1),0))return  true;if(D=n.Jl(t.i,A),D!=null){for(r=D,P=0,i=r.length;P<i;++P)if(e=r[P],e==5||e==6||e==7||e==8||e==15||e==16||e==33||e==34)return  true;return  false;}}return n.Ft(n.Tl(t.i,A));},n.yt=function(t,A){var e,r,P,i,D;if(t.i.K){if(pt(J(n.Ol(t.i,A),1),0))return  false;if(e=n.Jl(t.i,A),e!=null){for(P=e,i=0,D=P.length;i<D;++i)if(r=P[i],!n.qo(r))return  false;return  true;}}return n.qo(n.Tl(t.i,A));},n.zt=function(t,A){return n.fm(t.i,A)>1||n.im(t.i,A)==8;},n.At=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;for(v=0;v<t.e.length;v++)if(t.e[v]){for(k=false,r=U.xj(t.j,v),i=0,u=r.length;i<u;++i)A=r[i],t.a[A]|=H1,n.yt(t,A)&&(k=true);for(L=U.xj(t.k,v),T=0,g=L.length;T<g;++T)Q=L[T],t.c[Q]|=H1;if(t.f[v]){for(e=U.xj(t.j,v),P=0,D=e.length;P<D;++P)A=e[P],t.a[A]|=r0;for(l=U.xj(t.k,v),y=0,b=l.length;y<b;++y)Q=l[y],t.c[Q]|=r0;}if(k){for(e=U.xj(t.j,v),P=0,D=e.length;P<D;++P)A=e[P],t.a[A]|=yA;for(l=U.xj(t.k,v),y=0,b=l.length;y<b;++y)Q=l[y],t.c[Q]|=yA;}}for(c=new U.rrb(t.b);c.a<c.c.a.length;)Q=U.qrb(c).a,t.c[Q]=0;},n.Bt=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T;for(T=A.length,P=A,i=0,D=P.length;i<D;++i)r=P[i],y=t.a[r]&e1,(y==0||y>T)&&(t.a[r]&=fA,t.a[r]|=T);for(Q=e,l=0,L=Q.length;l<L;++l)u=Q[l],y=t.c[u]&e1,(y==0||y>T)&&(t.c[u]&=fA,t.c[u]|=T);},n.Ct=function(t,A,e){var r;for(r=U.xj(t.k,A).length;e>=r;)e-=r;for(;e<0;)e+=r;return e;},n.Dt=function(t,A){n.Et.call(this,t,A);},n.Et=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;this.i=t,this.g=7,this.j=new U.Mj(),this.k=new U.Mj(),this.b=new U.Mj(),this.a=B(o.Q7,H,6,this.i.f,15,1),this.c=B(o.Q7,H,6,this.i.g,15,1),this.i.$b(1),T=B(o.vfb,Tt,6,this.i.f,16,1),c=B(o.vfb,Tt,6,this.i.g,16,1);do for(D=false,e=0;e<this.i.f;e++)if(!T[e]){for(k=0,y=0;y<n.$o(this.i,e);y++)T[n.Zo(this.i,e,y)]||++k;if(k<2){for(T[e]=true,L=0;L<n.$o(this.i,e);L++)c[n._o(this.i,e,L)]=true;D=true;}}while(D);for(S=0;S<this.i.f&&T[S];)++S;if(S!=this.i.f){for(Q=B(o.Q7,H,6,this.i.f,15,1),Q[0]=S,g=B(o.Q7,H,6,this.i.f,15,1),g[0]=-1,u=B(o.Q7,H,6,this.i.f,15,1),u[S]=1,i=0,l=0,b=1;i<=l;){for(L=0;L<n.$o(this.i,Q[i]);L++)if(P=n.Zo(this.i,Q[i],L),P!=g[Q[i]]){if(u[P]!=0){n.$s(this,n._o(this.i,Q[i],L),T);continue;}T[P]||(u[P]=b,g[P]=Q[i],Q[++l]=P);}if(++i,i>l){for(e=0;e<this.i.f;e++)if(u[e]==0&&!T[e]){u[e]=++b,Q[++l]=e,g[e]=-1;break;}}}if((A&4)!=0&&(this.e=B(o.vfb,Tt,6,this.j.a.length,16,1),this.f=B(o.vfb,Tt,6,this.j.a.length,16,1),this.d=B(o.Q7,H,6,this.j.a.length,15,1),n.at(this,this.e,this.f,this.d,(A&8)!=0),n.At(this)),(A&2)!=0)for(r=0;r<this.i.g;r++)!c[r]&&n.fm(this.i,r)!=0&&(v=n.bt(this,r,T),v!=null&&n.Bt(this,v,n.jt(this,v)));}},n.Ft=function(t){return t==5||t==6||t==7||t==8||t==15||t==16||t==33||t==34;},x(151,1,{},n.Dt),s.g=0,o.C8=N(151),n.Gt=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;if(A==2||A==5)U.sj(t.F,n.ku(t.G,t.G.length)),t.b&&U.sj(t.a,n.lu(t.C,t.C.length));else if(A==4)T=n.St(n.ku(t.G,t.G.length)),U.Kxb(t.O,T)||(U.Ixb(t.O,T),U.sj(t.F,n.ku(t.G,t.G.length)),t.b&&U.sj(t.a,n.lu(t.C,t.C.length)));else if(A==3){if(T=n.St(n.ku(t.G,t.G.length)),!U.Kxb(t.O,T)){for(y=false,l=(u=new U.fxb(new U.lxb(new U.sqb(t.O.a).a).b),new U.Bqb(u));U.apb(l.a.a);){for(Q=(D=U.dxb(l.a),D.hd()),L=0,r=T,P=0,i=r.length;P<i;++P){for(e=r[P];L<Q.length&&Q[L]<e;)++L;if(L<Q.length&&e==Q[L]){y=true;break;}}if(y)break;}y||(U.Ixb(t.O,T),U.sj(t.F,n.ku(t.G,t.G.length)),t.b&&U.sj(t.a,n.lu(t.C,t.C.length)));}}else A==6&&(T=n.Tt(t,n.ku(t.G,t.G.length)),U.Kxb(t.O,T)||(U.Ixb(t.O,T),U.sj(t.F,n.ku(t.G,t.G.length)),t.b&&U.sj(t.a,n.lu(t.C,t.C.length))));},n.Ht=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c;if(u=n.$o(t.H,A),r=t.o[e],r>u)return  false;if(l=n.Ol(t.H,A),i=n.Ol(t.g,e),P=n.Jl(t.g,e),Q=n.Jl(t.H,A),pt(J(i,1),0)){if(P!=null){if(pt(J(l,1),0)){if(Q==null||!n.Yt(P,Q))return  false;}else if(Q!=null){if(n.$t(Q,P))return  false;}else if(n.Xt(n.Tl(t.H,A),P))return  false;}}else {if(pt(J(l,1),0))return  false;if(P!=null){if(Q!=null){if(!n.Yt(Q,P))return  false;}else if(!n.Xt(n.Tl(t.H,A),P))return  false;}else if(Q!=null||t.J[A]!=t.k[e])return  false;}if(pt(Pt(l,i),0)){if(pt(J(i,B1),0)){if(t.H.K&&S1(J(l,B1),0))return  false;if(r!=u)return  false;}if(pt(J(i,Z1),0)&&r>=u&&S1(J(l,Z1),0))return  false;}if(pt(J(t.I[A],tA(t.j[e])),0)||pt(J(t.B[e],tA(t.M[A])),0))return  false;if(D=J(i,EA),t.H.K){if(L=J(i,EA),nt(L,0)!=0&&(nt(D,0)==0||pt(J(D,tA(L)),0)))return  false;}else if(nt(D,0)!=0&&S1(J(D,t.M[A]),0))return  false;return n.Bl(t.g,e)!=0&&n.Bl(t.g,e)!=n.Bl(t.H,A)||n.Ml(t.g,e)!=0&&n.Ml(t.g,e)!=n.Ml(t.H,A)||n.Pl(t.g,e)!=0&&n.Pl(t.g,e)!=n.Pl(t.H,A)||(y=J(i,uP),nt(y,0)!=0&&n.xp(t.H,A)!=Jt(A0(y,47))-7)?false:(T=J(i,vA),nt(T,0)!=0?(T=A0(T,22),t.H.K?(c=Jt(A0(J(n.Ol(t.H,A),vA),22)),Jt(T)==c):n.Uo(t.H,A)==Jt(T)):true);},n.It=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T;if(l=t.K[A],D=t.n[e],(n.hm(t.g,e)&sn)!=0){if(Q=n.jm(t.H,A),P=n.jm(t.g,e),i=n.hm(t.g,e)&127,Q!=P&&!(Q==1&&(i&1)!=0)&&!(Q==2&&(i&2)!=0)&&!(Q==4&&(i&4)!=0)&&!(Q==32&&(i&32)!=0)&&!(Q==64&&(i&64)!=0)&&!(Q==16&&(i&16)!=0)&&!(Q==8&&(i&8)!=0))return  false;l&=-128,D&=-128;}if((l&~D)!=0)return  false;if(T=(n.hm(t.g,e)&iA)>>17,T!=0){if(t.H.K&&T==(n.hm(t.H,e)&iA)>>17)return  true;if(T<=2)return L=n.Yo(t.H,A),T==1?L>=8&&L<=12:L>=12;for(r=false,y=n.Ep(t.H),u=0;u<y.j.a.length;u++)if(U.xj(y.k,u).length==T&&n.rt(y,u,A)){r=true;break;}if(!r)return  false;}return  true;},n.Jt=function(t){var A,e;for(t.b=null,A=0;A<t.g.g;A++)n.Cm(t.g,A)&&(!t.b&&(t.b=new U.Mj()),e=new n.mu(),e.a=n.Zl(t.g,0,A),e.b=n.Zl(t.g,1,A),e.d=n._l(t.g,A),e.c=n.$l(t.g,A),U.sj(t.b,e));},n.Kt=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T;for(n.bw(t.g,t.N),Q=d.Math.max(t.g.f,t.g.g)+16,t.s=B(o.Q7,H,6,Q,15,1),t.u=B(o.Q7,H,6,Q,15,1),t.v=B(o.Q7,H,6,Q,15,1),t.t=B(o.vfb,Tt,6,Q+1,16,1),D=B(o.vfb,Tt,6,t.g.f,16,1),u=B(o.vfb,Tt,6,t.g.g,16,1),r=0,e=0;e<t.g.f;e++)if(!t.D[e]&&!D[e])for(t.s[r]=e,t.v[r]=-1,t.u[r]=-1,l=r;r<=l;){for(L=0;L<n.Mo(t.g,t.s[r]);L++)l=n.hu(t,r,l,L,D,u,-1);for(;t.t[++r];);}if(t.w=r,t.p!=0){for(l=t.w-1,i=0;i<t.d;i++)for(r=0;r<=l;){for(y=0;y<n.Mo(t.g,t.s[r]);y++)l=n.hu(t,r,l,y,D,u,i);for(;t.t[++r];);}for(A=0;A<t.g.f;A++)if(t.D[A]&&!D[A])for(t.s[r]=A,t.v[r]=-1,t.u[r]=-1,l=r;r<=l;){for(y=0;y<n.Mo(t.g,t.s[r]);y++)n.Zo(t.g,t.s[r],y)<t.g.f&&(l=n.hu(t,r,l,y,D,u,t.f[A]));for(;t.t[++r];);}for(t.e=B(o.Q7,H,6,t.d,15,1),T=0;T<t.d;T++)t.e[T]=-1;for(L=t.w;L<r;L++)P=t.f[t.s[L]],t.e[P]==-1&&(t.e[P]=L);}t.A=r;},n.Lt=function(t,A){var e,r,P,i,D,u,Q,l;for(l=n.Ep(t.H),u=0;u<l.j.a.length;u++)if(n.rt(l,u,A)){for(Q=B(o.Q7,H,6,2,15,1),P=0;P<2;P++)for(Q[P]=qt,r=n.Zl(t.H,P,A),i=0;i<n.$o(t.H,r);i++)e=n.Zo(t.H,r,i),e!=n.Zl(t.H,1-P,A)&&Q[P]>e&&(Q[P]=e);return D=0,n.qt(l,u,Q[0])&&++D,n.qt(l,u,Q[1])&&++D,D==2?2:D==1?1:2;}return 0;},n.Mt=function(t,A,e){var r,P,i,D,u;if(t.b){for(t.C=B(o.vfb,Tt,6,A.length,16,1),P=new U.rrb(t.b);P.a<P.c.a.length;)if(r=U.qrb(P),t.f==null||e==-1&&t.f[r.a]==-1&&t.f[r.b]==-1||e!=-1&&(t.f[r.a]==e||t.f[r.b]==e)){if(u=B(o.Q7,H,6,r.c+2,15,1),i=n.yp(t.H,u,t.G[r.a],t.G[r.b],r.c+1,A,null)-1,i<r.d||i>r.c)return  false;for(D=1;D<=i;D++)t.C[u[D]]=true;}}return  true;},n.Nt=function(t,A){var e,r,P,i,D,u,Q,l;for(P=0;P<t.g.g;P++)if((n.hm(t.g,P)&TA)!=0){if(i=n.gm(t.g,P),i==0)continue;if(e=n.Zl(t.g,0,P),r=n.Zl(t.g,1,P),t.f==null||A==-1&&t.f[e]==-1&&t.f[r]==-1||A!=-1&&(t.f[e]==A||t.f[r]==A)){if(D=t.G[e],u=t.G[r],Q=n.Xo(t.H,D,u),l=n.gm(t.H,Q),l==0&&(n.jq(t.H,Q)&&(l=n.Lt(t,Q)),l==0)||i==3||l==3)continue;if(n.Ut(t,P)==(i==l))return  false;}}return  true;},n.Ot=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g;for(D=0,Q=0;Q<t.g.f;Q++)if((t.f==null||t.f[Q]==A)&&pt(J(n.Ol(t.g,Q),L0),0)){if(T=t.G[Q],y=n.Nl(t.g,Q),b=n.Nl(t.H,T),y==0||y==3)continue;if(b==0||b==3)return  false;if(n.Hl(t.g,Q)==1){++D;continue;}if(n.Hl(t.H,T)==1)return  false;if(n.Hl(t.g,Q)==2){++D;continue;}if(n.Hl(t.H,T)==2||n.Zt(t,Q)==(y==b))return  false;}if(D!=0){for(P=B(o.Q7,H,6,D,15,1),i=0,l=0;l<t.g.f;l++)(t.f==null||t.f[l]==A)&&pt(J(n.Ol(t.g,l),L0),0)&&(y=n.Nl(t.g,l),y!=0&&y!=3&&(P[i++]=n.Gl(t.g,l)<<24|n.Hl(t.g,l)<<22|l));for(w.lAb(P,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),i=0;i<P.length;)for(L=P[i]&W1,c=t.G[L],e=P[i]&-4194304,r=n.Zt(t,L)^n.Nl(t.g,L)==n.Nl(t.H,c),++i;i<P.length&&(P[i]&-4194304)==e;i++)if(u=P[i]&W1,T=t.G[u],n.Hl(t.H,T)!=n.Hl(t.H,c)||n.Gl(t.H,T)!=n.Gl(t.H,c)||(g=n.Zt(t,u)^n.Nl(t.g,u)==n.Nl(t.H,T),g!=r))return  false;}return  true;},n.Pt=function(t,A,e){var r,P,i,D,u,Q,l;if(t.P=false,t.F=new U.Mj(),t.a=new U.Mj(),U.Iwb(t.O.a),!t.H||!t.g||t.g.f-t.p>t.H.f||t.g.g-t.q>t.H.g||t.g.f==0)return 0;for(A==6&&(t.N=63),n.du(t,e),r=B(o.vfb,Tt,6,t.H.f,16,1),t.G=B(o.Q7,H,6,t.g.f,15,1),U.Drb(t.G,-1),u=B(o.Q7,H,6,t.A,15,1),U.Jrb(u,u.length,-1),i=0;!t.P;){if(t.w!=0){if(l=t.u[i]==-1?t.H.f:n.Mo(t.H,t.G[t.u[i]]),++u[i],u[i]==l){if(u[i]=-1,i==0)break;--i,t.t[i]||(r[t.G[t.s[i]]]=false);continue;}if(t.u[i]==-1)r[u[i]]||n.Ht(t,u[i],t.s[i])&&(t.G[t.s[i]]=u[i],r[u[i]]=true,++i);else {if(n.Zo(t.H,t.G[t.u[i]],u[i])>=t.H.f)continue;P=n.Zo(t.H,t.G[t.u[i]],u[i]),t.t[i]?P==t.G[t.s[i]]&&n.It(t,n._o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&++i:r[P]||n.Ht(t,P,t.s[i])&&n.It(t,n._o(t.H,t.G[t.u[i]],u[i]),t.v[i])&&(r[P]=true,t.G[t.s[i]]=P,++i);}}if(i==t.w){if(n.Ot(t,-1)&&n.Nt(t,-1)&&n.Mt(t,r,-1)){for(Q=false,D=0;D<t.d;D++)if(n.Vt(t,r,u,D)){Q=true;break;}if(A==1&&!Q||!Q&&(n.Gt(t,A),A==2))return 1;}if(i==0)break;--i,t.t[i]||(r[t.G[t.s[i]]]=false);}}return t.F.a.length;},n.Qt=function(t,A){var e,r,P,i,D,u,Q,l;if(Q=0,t.K){switch(A<t.f&&n.vt(t.p,A)?Q=Pt(Q,sP):A<t.f&&n.ot(t.p,A)&&(Q=Pt(Q,2)),l=(P=t.u[A]&V0,P==0?0:P==f0?2:P==B1?3:4),l!=0&&(Q=Pt(Q,8),l>2&&(Q=Pt(Q,16)),l>3&&(Q=Pt(Q,32))),e=t.s[A],e<0?Q=Pt(Q,on):e>0&&(Q=Pt(Q,Qn)),D=t.j[A],D){case 0:break;case 1:Q=Pt(Q,r0);break;case 2:Q=Pt(Q,tn);break;case 3:Q=Pt(Q,iA);break;default:Q=Pt(Q,ln);}switch(r=n.Po(t,A),r){case 0:break;case 1:Q=Pt(Q,Cn);break;case 2:Q=Pt(Q,ze);break;case 3:Q=Pt(Q,Je);break;default:Q=Pt(Q,De);}u=t.o[A],u>0&&(Q=Pt(Q,N0)),u>1&&(Q=Pt(Q,LA));}else {switch(A<t.f&&n.vt(t.p,A)?Q=Pt(Q,sP):A<t.f&&n.ot(t.p,A)?Q=Pt(Q,2):Q=Pt(Q,4),(t.u[A]&q0)!=0?Q=Pt(Q,Ae):Q=Pt(Q,ne),l=(P=t.u[A]&V0,P==0?0:P==f0?2:P==B1?3:4),l==0?Q=Pt(Q,112):l==2?Q=Pt(Q,104):l==3?Q=Pt(Q,88):Q=Pt(Q,56),e=t.s[A],e==0?Q=Pt(Q,re):e<0?Q=Pt(Q,on):Q=Pt(Q,Qn),i=t.e[A]-n.tp(t,A)+n.op(t,A),i){case 0:Q=Pt(Q,1792);break;case 1:Q=Pt(Q,1664);break;case 2:Q=Pt(Q,1408);break;default:Q=Pt(Q,896);}switch(D=t.j[A],D){case 0:Q=Pt(Q,3932160);break;case 1:Q=Pt(Q,Pe);break;case 2:Q=Pt(Q,Mn);break;case 3:Q=Pt(Q,Rn);break;default:Q=Pt(Q,ln);}switch(r=n.Po(t,A),r){case 0:Q=Pt(Q,Oe);break;case 1:Q=Pt(Q,xe);break;case 2:Q=Pt(Q,Ne);break;case 3:Q=Pt(Q,Ge);break;default:Q=Pt(Q,De);}switch(u=t.o[A],u){case 0:Q=Pt(Q,98304);break;case 1:Q=Pt(Q,81920);break;default:Q=Pt(Q,49152);}}return Q;},n.Rt=function(t,A){var e;if(e=0,A<t.g&&(n.ut(t.p,A)||t.H[A]==8)||t.H[A]==8)e|=8;else switch(n.fm(t,A)){case 0:e|=16;break;case 1:e|=1;break;case 2:e|=2;break;case 3:e|=4;break;case 4:e|=32;break;case 5:e|=64;}return (t.F[A]&64)!=0?e|=256:t.K||(e|=128),A<t.g&&n.pt(t.p,A)?e|=j0:t.K||(e|=_0),e;},n.St=function(t){var A,e,r,P,i,D,u,Q,l,L;for(Q=0,r=t,i=0,u=r.length;i<u;++i)A=r[i],A==-1&&++Q;if(Q!=0)for(L=t,t=B(o.Q7,H,6,L.length-Q,15,1),l=0,e=L,P=0,D=e.length;P<D;++P)A=e[P],A!=-1&&(t[l++]=A);return w.lAb(t,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),t;},n.Tt=function(t,A){var e,r,P,i,D,u,Q,l;for(D=0,r=A,P=0,i=r.length;P<i;++P)e=r[P],e==-1&&++D;for(l=B(o.Q7,H,6,A.length-D,15,1),Q=0,u=0;u<A.length;u++)A[u]!=-1&&(l[Q]=n.mw(t.g,u)<<16|n.mw(t.H,A[u]),t.i!=null&&(l[Q]|=t.i[u]<<24),++Q);return w.lAb(l,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),l;},n.Ut=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c;for(i=false,P=0;P<2;P++)if(e=n.Zl(t.g,P,A),L=t.G[e],n.$o(t.H,L)>2){for(T=n.Zl(t.g,1-P,A),Q=qt,u=0;u<n.$o(t.g,e);u++)r=n.Zo(t.g,e,u),r!=T&&Q>r&&(Q=r);for(c=t.G[T],l=qt,D=0;D<n.$o(t.H,L);D++)y=n.Zo(t.H,L,D),y!=c&&l>y&&(l=y);t.G[Q]!=l&&(i=!i);}return i;},n.Vt=function(t,A,e,r){var P,i,D,u,Q,l,L,y;for(u=t.e[r],Q=u+1;Q<t.A&&t.f[t.s[Q]]==r;)++Q;for(L=u;L<Q;L++)e[L]=-1;for(D=u;;){if(y=t.u[D]==-1?t.H.f:n.Mo(t.H,t.G[t.u[D]]),++e[D],e[D]==y){if(e[D]=-1,D==u)break;--D,t.t[D]||(A[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);continue;}if(t.u[D]==-1)A[e[D]]||n.Ht(t,e[D],t.s[D])&&(t.G[t.s[D]]=e[D],A[e[D]]=true,++D);else {if(n.Zo(t.H,t.G[t.u[D]],e[D])>=t.H.f)continue;i=n.Zo(t.H,t.G[t.u[D]],e[D]),t.t[D]?i==t.G[t.s[D]]&&n.It(t,n._o(t.H,t.G[t.u[D]],e[D]),t.v[D])&&++D:A[i]||n.Ht(t,i,t.s[D])&&n.It(t,n._o(t.H,t.G[t.u[D]],e[D]),t.v[D])&&(A[i]=true,t.G[t.s[D]]=i,++D);}if(D==Q){if(n.Ot(t,r)&&n.Nt(t,r)&&n.Mt(t,A,r)){for(l=u;l<Q;l++)t.t[l]||(P=t.s[l],A[t.G[P]]=false,t.G[P]=-1);return  true;}--D,t.t[D]||(A[t.G[t.s[D]]]=false,t.G[t.s[D]]=-1);}}return  false;},n.Wt=function(t){return n.Pt(t,1,t.c)>0;},n.Xt=function(t,A){var e;for(e=0;e<A.length;e++)if(A[e]==t)return  true;return  false;},n.Yt=function(t,A){var e,r,P;for(P=0,r=0;r<t.length;r++){for(e=t[r];A[P]<e;)if(++P,P==A.length)return  false;if(A[P]>e)return  false;}return  true;},n.Zt=function(t,A){var e,r,P,i,D,u,Q,l;if(D=false,n.Qo(t.g,A)==0)for(i=1;i<n.$o(t.g,A);i++)for(u=0;u<i;u++)r=n.Zo(t.g,A,i),P=n.Zo(t.g,A,u),t.G[r]>t.G[P]^r>P&&(D=!D);else for(i=0;i<n.$o(t.g,A);i++){for(e=n.Zo(t.g,A,i),l=0,Q=B(o.Q7,H,6,3,15,1),u=0;u<n.$o(t.g,e);u++)Q[l]=n.Zo(t.g,e,u),Q[l]!=A&&++l;l==2&&t.G[Q[0]]>t.G[Q[1]]^Q[0]>Q[1]&&(D=!D);}return D;},n.$t=function(t,A){var e,r,P,i;for(P=0,i=0;P<t.length&&i<A.length;){if(e=t[P],r=A[i],e==r)return  true;e<r?++P:++i;}return  false;},n._t=function(t,A){var e,r,P,i,D;if(!A||A.q==0||!A.K){t.g=null;return;}for(t.g=A,t.r=false,n.bw(t.g,1),t.N=7,r=0;r<t.g.f;r++)pt(J(n.Ol(t.g,r),{l:L0,m:0,h:3}),0)&&(t.N=15);for(D=0;D<t.g.g;D++)(n.hm(t.g,D)&TA)!=0&&(t.N=15);for(t.L&&t.N!=7&&n.bw(t.H,t.N),t.p=0,t.q=0,t.D=B(o.vfb,Tt,6,t.g.f,16,1),P=0;P<t.g.f;P++)t.D[P]=n.Jm(t.g,P),t.D[P]&&++t.p;if(t.d=0,t.f=null,t.i=null,t.p!=0){for(i=0;i<t.g.g;i++)(t.D[n.Zl(t.g,0,i)]||t.D[n.Zl(t.g,1,i)])&&++t.q;for(e=0;e<t.g.q;e++)n.sn(t.g,e,t.D[e]);t.f=B(o.Q7,H,6,t.g.q,15,1),t.d=n.hp(t.g,t.f,true,false);}},n.au=function(t,A){t.i=A;},n.bu=function(t,A,e){n.cu(t,e),n._t(t,A);},n.cu=function(t,A){if(!A||A.q==0){t.H=null;return;}t.H=A,t.L=false,n.bw(t.H,1);},n.du=function(t,A){t.L||(n.gu(t,A),t.L=true),t.r||(n.eu(t,A),n.Kt(t),n.Jt(t),t.r=true);},n.eu=function(t,A){var e,r,P,i,D,u,Q,l,L,y;for(i=null,Q=null,D=null,n.bw(t.g,t.N),t.o=B(o.Q7,H,6,t.g.f,15,1),r=0;r<t.g.f;r++)t.o[r]=n.$o(t.g,r);if(t.p!=0){for(l=new n.uw(t.g.q,t.g.r),y=B(o.vfb,Tt,6,t.g.q,16,1),P=0;P<t.g.q;P++)y[P]=!t.D[P];for(n.Co(t.g,l,y,true,null),n.bw(l,t.N),n.fu(t,l,A),i=t.j,Q=t.n,D=t.k,L=0,e=0;e<t.g.f;e++)t.D[e]||(t.o[e]=n.$o(l,L++));}if(n.fu(t,t.g,A),t.p!=0){for(L=0,e=0;e<t.g.q;e++)t.D[e]||(t.j[e]=i[L],t.k[e]=D[L++]);for(L=0,u=0;u<t.g.r;u++)!t.D[n.Zl(t.g,0,u)]&&!t.D[n.Zl(t.g,1,u)]&&(t.n[u]=Q[L++]);}},n.fu=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;for(g=A.f,t.j=B(o.R7,e0,6,A.f,14,1),t.k=B(o.Q7,H,6,A.f,15,1),u=0;u<g;u++)t.j[u]=en(J(Pt(n.Qt(A,u),A.B[u]),UP),W8),t.k[u]=A.C[u],(e&1)!=0&&(t.k[u]+=A.s[u]+16<<8),(e&2)!=0&&(t.k[u]+=A.A[u]<<16);for(t.B=B(o.R7,e0,6,A.f,14,1),v=(n.bw(A,7),A.p),b=0;b<v.j.a.length;b++){for(c=false,l=U.xj(v.k,b),L=0,y=l.length;L<y;++L)if(T=l[L],(A.G[T]&sA)!=0){c=true;break;}if(!c)for(S=U.xj(v.k,b).length,P=U.xj(v.j,b),i=0,D=P.length;i<D;++i)r=P[i],S==3?t.B[r]=Pt(t.B[r],qA):S==4?t.B[r]=Pt(t.B[r],VA):S==5?t.B[r]=Pt(t.B[r],IA):S==6?t.B[r]=Pt(t.B[r],OA):S==7&&(t.B[r]=Pt(t.B[r],xA));}for(k=A.g,t.n=B(o.Q7,H,6,A.g,15,1),Q=0;Q<k;Q++)t.n[Q]=(n.Rt(A,Q)|A.G[Q])&6291967^6291840,(e&4)!=0?(t.n[Q]&2)!=0&&(t.n[Q]|=8):(e&8)!=0&&(t.n[Q]&2)!=0&&Q<A.g&&n.pt(A.p,Q)&&(t.n[Q]|=8);},n.gu=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c;for(n.bw(t.H,t.N),L=t.H.f,t.J=B(o.Q7,H,6,L,15,1),t.I=B(o.R7,e0,6,L,14,1),D=0;D<L;D++)t.I[D]=en(J(Pt(n.Qt(t.H,D),n.Ol(t.H,D)),UP),W8),t.J[D]=n.Tl(t.H,D),(A&1)!=0&&(t.J[D]+=n.Bl(t.H,D)+16<<8),(A&2)!=0&&(t.J[D]+=n.Ml(t.H,D)<<16);for(t.M=B(o.R7,e0,6,L,14,1),T=n.Ep(t.H),l=0;l<T.j.a.length;l++)for(c=U.xj(T.k,l).length,r=U.xj(T.j,l),P=0,i=r.length;P<i;++P)u=r[P],c==3?t.M[u]=Pt(t.M[u],qA):c==4?t.M[u]=Pt(t.M[u],VA):c==5?t.M[u]=Pt(t.M[u],IA):c==6?t.M[u]=Pt(t.M[u],OA):c==7&&(t.M[u]=Pt(t.M[u],xA));for(e=0;e<L;e++)c=n.Uo(t.H,e),c==0?t.M[e]=Pt(t.M[e],lA):c>7&&(t.M[e]=Pt(t.M[e],un));for(y=t.H.g,t.K=B(o.Q7,H,6,y,15,1),Q=0;Q<y;Q++)t.K[Q]=(n.Rt(t.H,Q)|n.hm(t.H,Q))&6422527^6291840;},n.hu=function(t,A,e,r,P,i,D){var u,Q;return u=n.Zo(t.g,t.s[A],r),u<t.D.length&&(!t.D[u]||t.f[u]==D)&&u!=t.u[A]&&(Q=n._o(t.g,t.s[A],r),!i[Q]&&!n.Cm(t.g,Q)&&(t.s[++e]=u,t.u[e]=t.s[A],t.v[e]=Q,i[Q]=true,P[u]?t.t[e]=true:P[u]=true)),e;},n.iu=function(){this.c=8,this.O=new U.Oxb(new Lt.G4());},n.ju=function(t){this.c=t,this.O=new U.Oxb(new Lt.G4());},n.ku=function(t,A){var e;return e=B(o.Q7,H,6,A,15,1),h.Qkb(t,0,e,0,d.Math.min(t.length,A)),e;},n.lu=function(t,A){var e;return e=B(o.vfb,Tt,6,A,16,1),h.Qkb(t,0,e,0,d.Math.min(t.length,A)),e;},x(64,1,{},n.iu,n.ju),s.c=0,s.d=0,s.p=0,s.q=0,s.r=false,s.w=0,s.A=0,s.L=false,s.N=0,s.P=false,o.F8=N(64),n.mu=function(){},x(215,1,{215:1},n.mu),s.a=0,s.b=0,s.c=0,s.d=0,o.D8=N(215),n.pu=function(){n.pu=f1,n.nu=j(F(o.Ocb,1),yt,2,6,["QM@HzAmdqjF@","RF@Q``","qC`@ISTAlQE`","`J@H","QM@HzAmdqbF@","qC`@ISTAlQEhqPp@","sJP@DiZhAmQEb","RF@QPvR@","QM@HzA@","qC`@ISTAlQEhpPp@","qC`@Qz`MbHl","sJP@DiZhAmQEcFZF@","RFPDXH","qC`@IVtAlQE`","QM@HvAmdqfF@","sGP@DiVj`FsDVM@","`L@H","sJP@DizhAmQEcFBF@","sJP@DjvhAmQEb","sFp@DiTt@@AlqEcP","sGP@LdbMU@MfHlZ","QMHAIhD","QM@HzAy@","sJP@DkVhAmQEb","sNp@DiUjj@[\\QXu`","sJP@DiZhAmQEcFBF@","sGP@DjVj`FsDVM@","RFPDTH","RG@DXOH@","sGP@Divj`FsDVMcAC@","sGP@Dj}j`FsDVM@","qC`@Qz`MbHmFRF@","sNp@LdbJjj@[\\QXu`","QMHAIhGe@","QM@HzAyd`","QM`AIhD","qC`@ISTA@","sGP@DkUj`FsDVM@","qC`@IVtAlQEhqPp@","sNp@DiUjj@[\\QXuqea`@","KAx@@IRjuUPAlHPfES\\","QM`BN`P","sJP@DjZhAmQEcFJF@","Hid@@DjU^nBBH@FtaBXUMp`","sNp@Diujj@[\\QXuq`a`@","sJP@DjvhAmQEcFZF@","sJP@DjZhAmQEcFFF@","sOp@DjWkB@@FwDVM\\YhX@","sNp@Dj}Zj@[\\QXu`","sNp@DiWjj@[\\QXuq`a`@","sOp@DjWkB@@D","KAx@@ITouUPAlHPfES\\","KAx@@YIDTjjh@vDHSBin@","sNp@DkUZj@[\\QXu`","RFPDXOH@","QM`BN`^L`","qC`@ISTAy@","sGP@LdbMU@MfHl[FVF@","qCb@AIZ`H","KAx@@IRjuUPAlHPfES]FFa`@","KAx@@ITnuUPAlHPfES\\","HiD@@DiUVjj`AmHPfES\\H","sNp@DjUjj@[\\QXu`","sJP@DkVhAmQEcFJF@","sGP@DjVj`FsDVMcCC@","qC`@Qz`MbHmFBF@","sJP@DkfhAmQEb","qC`@IVtAlQEhsPp@","sGP@Djuj`FsDVM@","sGP@Dj}j`FsDVMcMC@","sJP@DiZhA@","KAx@@ISjuUPAlHPfES]F@a`@","sJP@DjZhAmQEcFRF@","KAx@@IRnuUPAlHPfES]F@a`@","HiD@@DjWvjj`AmHPfES\\H","QMHAIhGd@","sNp@DiUjj@[\\QXuq`a`@","KAx@@IVjmUPAlHPfES\\","sGP@DjVj`FsDVMcMC@","QM`AIhGe@","HiD@@LdbJRjjh@[RDIaTwB","qCp@AIZ`H","sGP@LdbMU@MfHl[FFF@","QMDARVA@","sNp@LdbJjj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqca`@","sGP@Dkej`FsDVM@","qCb@AIZ`OI@","HaD@@DjUZxHH@AlHPfES]FLa`@","sGP@DkYj`FsDVM@","qCb@AIV`H","sNp@LdbJjj@[\\QXuqea`@","sGP@DkUj`FsDVMcEC@","sFp@DiTt@@Axa@","Hmt@@DjU_ZxHHj@AmhPfES\\Lj","QM`BN`^P","qCb@AIZ`OH`","sFp@DiTt@@AxaP","sGP@Djuj`FsDVMcEC@","sGP@Djuj`FsDVMcIC@","sGP@DkUj`FsDVMcKC@","sJP@DkfhAmQEcFRF@","sGP@DjVj`FsDVMcIC@","HaD@@DjUZxHH@AlHPfES]FFa`@","qC`@IRtDVqDV@","sNp@Dj}Zj@[\\QXuqfa`@","KAx@@ITnuUPAlHPfES]FFa`@","HiD@@DkUUjj`AmHPfES\\H","sJQ@@dkU@H","qC`@Qz`H","KAx@@IUkmUPAlHPfES\\","KAx@@ITouUPAlHPfES]FJa`@","sJP@H~j@[TQX`","sGP@DjZj`FsDVM@","sJP@DkVhAmQEcFFF@","sJX@@eKU@H","sJP@DizhAy@","QMHAIhGbP","KAx@@ITouUPAlHPfES]FNa`@","HaD@@DjUZxHD@AlHPfES\\","HaD@@DjUZxHH@A@","sNp@LdbJjj@[\\QXuqaa`@","Hed@@LdbRQUUUP@vTHSBinFP","KAx@@ITouUPAlHPfES]FLa`@","sNp@DkUZj@[\\QXuqba`@","KAx@@ITjuUPAlHPfES]FNa`@","KAx@@YIDTjjh@vDHSBincGPp@","HaD@@DjYvxH`@AlHPfES]FLa`@","RF@QP`","qCb@AIj`H","sNp@DjUjj@[\\QXuqaa`@","sNp@DkVZj@[\\QXu`","KAx@@YIDUJjh@vDHSBin@","sGP@DkYj`FsDVMcIC@","sGP@DjVj`FsDVMcAC@","sGP@DiVj`D","sJP@DkVhAmQEcFZF@","sNp@LdbLjj@[\\QXu`","QM@HvAmdqbF@","HaD@@DjWjXHB@AlHPfES\\","sNp@DjwZj@[\\QXuqba`@","sNp@LdbJjj@[\\QXuqda`@","sFp@DiTt@@Axa`","HiD@@Djuujj`AmHPfES\\H","sNp@DkUZj@[\\QXuqca`@","sJP@DiZhAy@","KAx@@YIDTjjh@vDHSBincCPp@","KAx@@IWNmUPAlHPfES\\","KAx@@IVkMUPAlHPfES\\","sJQ@@dju@H","qCb@AIZ`OH@","qC`@ISTAxa@","sNp@DjyZj@[\\QXu`","Hid@@DjUfaBB`@FtaBXUMp`","HiD@@DiUVjj`AmHPfES\\LXBF@","KAx@@IUjmUPAlHPfES\\","HiD@@DjWvjj`AmHPfES\\LXjF@","sJP@DjVhAmQEb","qCb@AIV`OH`","HiD@@LdbJRjjh@[RDIaTwCFDa`@","KAx@@YIDTjjh@vDHSBinc@Pp@","sNp@DjUjj@[\\QXuqda`@","qC`@Qz`OED","sJP@DkfhAmQEcFZF@","KAx@@YIDbjjh@vDHSBincDPp@","sGP@Djyj`FsDVMcMC@","KAx@@IVrmUPAlHPfES\\","qCp@AIZ`OI@","sJX@@dkU@H","sJQ@@dkU@OH`","sNp@Di]ZjBBvxbqk@","Hkl@@DjU_Uk``bj`@[VDIaTwCJzX","sGP@DjZj`FsDVMcEC@","Hid@@DjU^nBBH@FtaBXUMpqcHX@","sNp@DkeZj@[\\QXu`","sNp@DjYjj@[\\QXuqca`@","sGQ@@djuT@`","HiD@@LdbJTjjh@[RDIaTwB","sOp@DjWkB@@Gd`","HeT@@LdbbRKBDQD@CYPaLJfxY@","qCr@XIKTA@","HiD@@DjW^jj`AmHPfES\\LXJF@","HeT@@DjU]k``b`@[JDIaTwCH","sGP@Djuj`FsDVMcCC@","`IH`B","sOp@DjWkB@@GdX","sJQ@@eKU@H","KAx@@YIDUJjh@vDHSBincBPp@","sJX@@eKU@OH@","KAx@@YIDTjjh@vDHSBincAPp@","sOq@@drm\\@@@`","KAx@@IUkMUPAlHPfES\\","qCp@AIj`H","Hed@@DjUUjjj@FraBXUMpr","sGX@@eJuT@`","sGP@DkUj`FsDVMcCC@","HiD@@Dj}Ujj`AmHPfES\\LXrF@","KAx@@ITouUPAlHPfES]FHa`@","Hed@@DjWujjj@FraBXUMpsFIa`@","sGP@DiUj``mfHlZ","sFp@DiTvjhAlqEcP","Hid@@DjU^nBBH@FtaBXUMpq`XX@","sJP@DkVdAmQEb","qCp@AIZ`OH`","QMhDRVA@","qC`@ISJAlQE`","qCp@BOTAyhl","sJX@@eOU@ODB","sFp@DiTt@@AyaB","sGP@DkUj`FsDVMcMC@","Hid@@DjYUaBH`@FtaBXUMpqcHX@","qC`@Qz`OH@","HiD@@DjUVjj`AmHPfES\\LXZF@","sJP@H~j@[TQXqda`@","sJX@@eKU@OI@","sNp@Djejj@[\\QXu`","sJQ@@dsU@H","sJQ@@dkU@OI`","KAx@@YIMDVjh@vDHSBin@","Hid@@DjU^nBBD@FtaBXUMp`","sNp@DkgZj@[\\QXuqca`@","qC`@IRtDVqDVcEC@","Hed@@LdbRQeUUP@vTHSBinFP","sNp@DiUjj@P","qC`@IRtDT","sNp@DkYZj@[\\QXuqca`@","KAx@@IUkmUPAlHPfES]FDa`@","KAx@@IVjmUPAlHPfES]FNa`@","sOx@@drm\\@@@`","KAx@@ITjuUPAlHPfES]FBa`@","QMDARVAyH","sJP`@dfvhA@","HeT@@DjU_k``b`@[JDIaTwCLXfF@","KAx@@IToUUPAlHPfES]FJa`@","sGP@DkYj`FsDVMcEC@","qCb@AIZ`ODH","`I@`B","KAx@@IUzmUPAlHPfES]FFa`@","sNp@DkfZj@[\\QXu`","KAx@@ITnuUPAlHPfES]F@a`@","HiD@@LddURjjh@[RDIaTwB","sNp@Dj~Zj@[\\QXuqfa`@","Hed@@Dj{uZjj@FraBXUMpr","KAx@@ITsUUPAlHPfES\\","Hid@@LdbRQk``b@AmHPfES\\LXrF@","sOp@DjWkB@@GdH","sJQ@@dkU@OH@","Hid@@DjU^nBBH@FtaBXUMpqahX@","sGP@DiYj``mfHlZ","KAx@@IToUUPAlHPfES]FLa`@","qCp@AJZ`ODH","Hmt@@DjU]ZxHHj@AmhPfES\\Lj","sGP@DkUjPFsDVM@","qC`@IVtA@","Hed@@LdbJReUUP@vTHSBinFP","sNp@DjuZj@[\\QXuqea`@","KAx@@IUkmUPAlHPfES]FNa`@","HiD@@DkVUjj`AmHPfES\\H","Hed@@DkUeZjj@FraBXUMpr","sNp@DkVZj@[\\QXuqea`@","sJP@DiVhHKZbKFLLL@","HiD@@Djuyjj`AmHPfES\\H","sNp@DjUjj@[\\QXuq`a`@","HeT@@DjYUXPbH`@[JDIaTwCH","HiD@@DjwUjj`AmHPfES\\LXRF@","sNq@@djmUPB","KAx@@YIEEZjh@vDHSBincCPp@","sGP@Di^V`dmfHlZ","Hid@@DjYUaBHP@FtaBXUMp`","sNp@DjYjj@[\\QXuqba`@","sGP@Dkej`FsDVMcKC@","HeT@@DjU^k``b`@[JDIaTwCH","qC`@Qv`MbHmFBF@","sGQ@@djmT@`","qCr@XIKTAyH","qC`@IVtAlQEhpPp@","Hid@@LdbbQxXF@@AmHPfES\\LXjF@","sGP@DkYj`FsDVMcCC@","KAx@@IVsMUPAlHPfES\\","qCp@AIj`ODl","HiD@@DkeUjj`AmHPfES\\H","deT@@DjU[k``b`@vTHSBinFP","sJP@DkVdAmQEcFRF@","HiD@@LdbJTjjh@[RDIaTwCFDa`@","HiD@@DkYyjj`AmHPfES\\H","sJP@DjZhAyH","KAx@@IVkMUPAlHPfES]FDa`@","sJX@@dkU@OI@","Hed@@LdbRQUUUP@vTHSBinFXpLL@","Hed@@DjuUZjj@FraBXUMpr","sGP@Djfj`FsDVMcKC@","sNp@DkVZj@[\\QXuqba`@","sNp@DjyZj@[\\QXuqfa`@","qCb@AIj`OH@","sNp@DjUZj@[\\QXu`","KAx@@IWOMUPAlHPfES\\","Hid@@DjU^nBBH@D","Hed@@DjuvZjj@FraBXUMpr","sJP@DiVhHKZbKFLtL@","dmt@@DjU_ZxHHj@C[PaLJfxYVLKC@","sNp@DjuZj@[\\QXuqca`@","sJP@DkfhAmQEcFJF@","sNp@LdbJZj@[\\QXu`","HeT@@DjU_k``b`@[JDIaTwCLXFF@","KAx@@IVlmUPAlHPfES]FNa`@","HeT@@LdbbRKBDQD@CYPaLJfxYcEPp@","Hid@@DjUZnBBH@FtaBXUMpqcHX@","qCa@CIKTA@","HiD@@Dj~]jj`AmHPfES\\LXFF@","sKP@Di\\Zj@[TQX`","sGP@Djfj`FsDVMcEC@","HiD@@DkgYjj`AmHPfES\\H","sNp@DjuZj@[\\QXuqaa`@","KAx@@YIMDVjh@vDHSBincDPp@","sJP@DjVhHKZbKFLTL@","Hid@@LdbRQk``b@AmHPfES\\LXZF@","HiD@@Dj}Ujj`AmHPfES\\LXzF@","HeT@@DjU_k``bP@[JDIaTwCH","sNp@DkUZi@[\\QXu`","HiD@@DjYfjj`AmHPfES\\H","sGP@DjZj`FsDVMcAC@","Hmt@@DjU_jxHHj@AmhPfES\\Lj","Hid@@LdbRQk``R@AmHPfES\\H","KAx@@YIDUJjh@vDHSBincDPp@","qCr@XIKTAyD","sOq@@drm\\@@@|`@","Hed@@DjW^jjj@FraBXUMpsFBa`@","HeT@@DjY]zXFB@@[JDIaTwCH","Hkl@@DjU_Vk``bj`@[VDIaTwCJzX","Hid@@DjY}nBHH@FtaBXUMpqcHX@","sGX@@eKuT@|d@","sGP@Dj^Y`FsDVM@","HcL@@DjU_ZnBBJh@FqaBXUMprn`","sJP@DkVdAmQEcFJF@","sOq@@drm\\@@@|b@","sNp@DjyZj@[\\QXuqaa`@","HaD@@DjUZxHH@AyD@","qC`@Qv`H","dmt@@DjU_ZxHHj@C[PaLJfxYVLYC@","sGP@Dkej`FsDVMcMC@","Hed@@DjUUjjj@FraBXUMpsFHa`@","HeT@@LdbbRkBDQD@CYPaLJfxY@","KAx@@IU{MUPAlHPfES]FLa`@","RG@DTH","sJY@DDeVhA@","KAx@@YIDUJjh@vDHSBinc@Pp@","sJX@@dkU@OI`","sJQ@@dju@OI`","HeT@@LdbbRKBDQD@CYPaLJfxYcFPp@","sFp@DiTvjhAlqEcXpPp@","HaD@@DjUZxHH@AyG@","sNx@@eJ}UPB","sNp@LddUjj@[\\QXuqca`@","HaDH@@RVU[j@@@D","sNp@DkgZi@[\\QXu`","sGY@LDeVj`D","sNp@LdbJfZBZvxbqk@","sJP`@dfvhAyL","sGX@AddQjhAxe`","Hmt@@DjU_ZxHHj@AmhPfES\\LkFIa`@","qCh@CIKTA@","sNp@LdbLjj@[\\QXuq`a`@","sOq@@drm\\@@@|a@","KAx@@IUzmUPAlHPfES]FJa`@","sNx@AddQUUPB","sGP@Di]jP`mfHlZ","sJP`@TeZhA@","KAx@@IRjmUPHKXPaLJfx","HeT@@LdbRTM\\DDT@CYPaLJfxY@","HaF@@@Rfu[j@@@D","Hid@@DjYUaBH`@FtaBXUMpqchX@","KAx@@IUjmTpAlHPfES\\","Hid@@DjU^nBBD@FtaBXUMpqcHX@","sGP@DiUj``mfHl[FFF@","KAx@@IUvmUPAlHPfES]FLa`@","Hed@@LdbQTUUUP@vTHSBinFXqDL@","sJP@DkVhA@","sOx@@drm\\@@@|b@","KAx@@IUkMUPAlHPfES]FDa`@","HeT@@LdbRQU\\DDT@CYPaLJfxY@","HiD@@Dj}Yjj`AmHPfES\\LXrF@","HiD@@Dj{ujj`AmHPfES\\LXFF@","KAx@@IWNmUPAlHPfES]FFa`@","KAx@@IRkMUPHKXPaLJfx","sJP@DjYdAmQEcFZF@","sJY@LDeZhAyL","HaDH@@RVU[f@@@D","sJP`@deVhAyB","HaD@@DjWjZjj`AlHPfES\\","sGP@DkYj`FsDVMcMC@","sNp@DkgZj@[\\QXuqea`@","sJQ@@dlu@H","HeT@@DjU]k``b`@[JDIaTwCLXrF@","sJX@@dkU@OH`","RFDDQFCr`","sJP@DiYXIKZbKFLLL@","KAx@@YIHjjjh@vDHSBincGPp@","Hk\\@@DjU^ukmLHH@@@AmXPfES\\Lki`","sGQ@@djmT@|b@","Hid@@DjUfaBB`@FtaBXUMpqahX@","sNx@@eRmUPB","Hmt@@LdbRVak``ah@FvaBXUMprh","qCr@XIJtA@","KAx@@IWMmUPAlHPfES]FNa`@","HeT@@DjYYZPbJ@@[JDIaTwCH","sNp@DkfZj@[\\QXuqea`@","Hid@@DjU^nBAHAEVtaBXUMp`","dmt@@DjUgZDHJJ@C[PaLJfxYT","sGP@DkejPFsDVM@","sNx@@eJmUPB","qCb@AIf`H","HcL@@DjU_VnBBJh@FqaBXUMprnqcXX@","Hid@@DjUZnBBH@FtaBXUMpqahX@","sNp@LdbQZjBBvxbqkcGC@","sOx@@drm\\@@@|c@","sJP@H~j@^R@","KAx@@YIDcFjhDElHPfES\\","Hid@@DjUZnBAH@FtaBXUMp`","sNp@LddUji@[\\QXu`","sGP@DjfjPFsDVM@","HeT@@DjYUXPbD`@[JDIaTwCH","KAx@@IUoMUPAlHPfES]FDa`@","sFp@DiTt@@AyaD","Hed@@DjuuZjj@FraBXUMpsFIa`@","HeT@@DjUghP`h`@[JDIaTwCLXfF@","sOp@DjWkjj`FwDVM\\YhX@","sGP@Djfj`FsDVMcIC@","KAx@@IRkmUPHKXPaLJfzL]C@","sNx@@djmUPB","QM`AIdD","sOp@DjWkB@@Gbe@","sNp@DjyZj@[\\QXuqca`@","QM@HuAmd`","sNp@LddUjj@[\\QXuqea`@","daD@@DjUZxHD@CXPaLJfzLYC@","qCb@AIZPH","HiD@@LdbJTjjh@[RDIaTwCF@a`@","Hmt@@DjU_ZxHHi@AmhPfES\\Lj","HaDH@@RYWih@H@D","HiD@@LdbJTjjh@[RDIaTwCFHa`@","sGX@@djuT@|a@","sNp@DkfZj@[\\QXuqaa`@","Hid@@DjU^nBBH@GdL","KAx@@IVkMUPAlHPfES]FJa`@","qCr@XIKTAy@","HmT@@Dj{uVjjh@[ZDIaTwCJqaXX@","dmt@@DjYWVDHbJ@C[PaLJfxYT","Hif@@@RUe^Fh@@@P","HaDH@@Rfu[j@@@GdH","KAx@@IVsMUPAlHPfES]FDa`@","sKP@Di\\Zj@[TQXq`a`@","sJX@@eMU@OH@","HeT@@DjU^k``b`@[JDIaTwCLXFF@","Hmt@@LdbbRJXPbHh@FvaBXUMprh","sJP@DjvhAmQEcFBF@","dmt@@LdbbRNXPbHh@MmBDpj[aeXplL@","sJP`@dfvhAyD","sGP@Di^V`dmfHl[FVF@","KAx@@IVsmUPAlHPfES]FBa`@","sOq@@drm\\@@@|PP","sJY@BDeZhA@","HeT@@LdbRbmBDED@CYPaLJfxY@","Hed@@Djy[Zjj@FraBXUMpr","HeT@@DjU]k``b`@[JDIaTwCLXFF@","Hid@@DjUfaBB`@D","qCa@CIJtA@","QMPARVA@","Hid@@DjUfaBB`@FtaBXUMpqcHX@","sJY@BDfZhA@","HeT@@DjUghP`hP@[JDIaTwCH","Hed@@Dj{uZjj@FraBXUMpsFIa`@","dmt@@DjYUvDHbJ@C[PaLJfxYVLUC@","sNp`@dfuZj@P","sJQ@@dmU@OH@","sJX@@dmU@H","HeT@@DjU]k``b`@[JDIaTwCLXZF@","HiD@@LdfbJZjh@[RDIaTwCFAa`@","sOx@@drm\\@@@|a@","deT@@DjUfhP`h`@vTHSBinFP","dmt@@DjU]ZxHHj@C[PaLJfxYVLKC@","sOp@DjWkjj`FwDVM\\XHX@","HcL@@LdbbRNSBDQEP@McBDpj[ae]cFpp@","HiD@@Dj}Yji`AmHPfES\\H","HaDH@@RYe[hB@@D","Hid@@DjU^njjj@FtaBXUMpq`XX@","deT@@LdbbRKBDQB@FraBXUMpr","QMPARZA@","sOq@@drm\\@@@|QX","HaD@@DjYvxH`@A@","HcL@@LdbbRNcBDQEP@McBDpj[ae]@","QMhDRZA@","RG@DXLHmP","QM`BN`XQYd","RG@DTLHmP","QMHAIXFEVd","QMDARVAaH","RFPDXLHmP","RF@Q`vRbdLEC@","RF@QpvR@","QO@HyjAmd`","`II@B","`II@CFspqJp","`II@CF[@hM@prB`","`H@[T[|B`XN@PdM@p|@bHrBcDk@","RG@DXMj}F@","QM`BN`[L~b@","RG@DTMj}D@","QMHAIXFt~j@","QMDARVA}L@","RFPDXMj}D@","sKP@Di\\YZ@[TQXqaa`@","eF`BHD"]);},n.qu=function(t,A){var e,r;if(!A)return null;for(r=B(o.Q7,H,6,(n.nu.length+31)/32|0,15,1),A=n.vu(A),n.cu(t.g,A),e=0;e<n.nu.length;e++)n._t(t.g,n.ou[e]),n.Pt(t.g,1,4)>0&&(r[e/32|0]|=1<<31-e%32);return r;},n.ru=function(t,A){var e,r;if(!A)return null;for(r=B(o.R7,e0,6,(n.nu.length+63)/64|0,14,1),A=n.vu(A),n.cu(t.g,A),e=0;e<n.nu.length;e++)n._t(t.g,n.ou[e]),n.Pt(t.g,1,4)>0&&(r[e/64|0]=Pt(r[e/64|0],h1(1,63-e%64)));return r;},n.su=function(){var t,A;if(n.ou==null)for(A=new n._q(false),n.ou=B(o.Q8,GA,26,n.nu.length,0,1),t=0;t<n.nu.length;t++)n.ou[t]=n.Pq(A,n.nu[t]),n.bw(n.ou[t],1);},n.tu=function(t){var A;if(t.f!=null){for(A=0;A<t.f.length;A++)if((t.c[A]&~t.f[A])!=0)return  false;}else return  false;return n.uu(t);},n.uu=function(t){return !t.d&&(t.d=n.Rq(new n._q(false),t.e)),!t.a&&(t.a=n.Rq(new n._q(false),t.b)),n.cu(t.g,t.d),n._t(t.g,t.a),n.Wt(t.g);},n.vu=function(t){var A,e;if(t.K){for(A=0;A<t.q;A++)if(pt(J(t.B[A],i0),0)){for(t=new n.vw(t),e=A;e<t.q;e++)pt(J(t.B[e],i0),0)&&(t.C[e]=-1);n.vl(t);}}return t;},n.wu=function(t,A,e){t.b=null,t.a=A,e==null?t.c=n.qu(t,A):t.c=e;},n.xu=function(t,A,e){t.e=null,t.d=A,e==null?t.f=n.qu(t,A):t.f=e;},n.yu=function(){n.pu(),this.g=new n.iu(),n.su();},n.zu=function(t){n.pu();var A,e,r,P,i,D;if(t==null)return null;for(A=B(o.M7,Y,6,t.length*8,15,1),r=0;r<t.length;r++)for(D=t[r],P=7;P>=0;P--)e=D&15,e>9&&(e+=7),A[r*8+P]=48+e<<24>>24,D>>=4;return h.$jb((i=A,q1.Cnb(),i));},n.Au=function(t){n.pu();var A,e,r,P;if(w.SAb(t).length==0||(w.SAb(t).length&7)!=0)return null;for(r=B(o.Q7,H,6,w.SAb(t).length/8|0,15,1),e=0;e<w.SAb(t).length;e++)P=e/8|0,A=(w.NAb(e,w.SAb(t).length),w.SAb(t).charCodeAt(e)-48),A>16&&(A-=7),r[P]<<=4,r[P]+=A;return r;},n.Bu=function(t,A){n.pu();var e,r,P,i;for(i=0,r=0,P=0,e=0;e<t.length;e++)i+=rn(t[e]&A[e]),r+=rn(t[e]),P+=rn(A[e]);return i/d.Math.sqrt(r*P);},n.Cu=function(t,A){n.pu();var e,r,P;for(P=0,e=0,r=0;r<t.length;r++)P+=rn(t[r]&A[r]),e+=rn(t[r]|A[r]);return P/e;},x(180,1,{},n.yu),o.E8=N(180),n.Eu=function(t){t.j=1,t.k=10,t.n=400,t.f=400,t.i=true,t.d="black",t.b=new U.Mj(),t.a=new U.Mj(),t.c=new h.Kkb(),t.e=new j1.Jgb(12);},n.Fu=function(t,A){var e,r;return e=(r=j1.Hgb(t.e,A),new B0._gb(r)).b,e;},n.Gu=function(t){return t.k;},n.Hu=function(t){var A,e,r,P,i,D;if(i='<svg id="'+n.Ou(t.g!=null?t.g:"mol"+n.Du)+Y1+'xmlns="http://www.w3.org/2000/svg" version="1.1" width="'+t.n+'px" height="'+t.f+'px" viewBox="0 0 '+t.n+" "+t.f+`">
`,D=t.i?"<style> #"+n.Ou(t.g!=null?t.g:"mol"+n.Du)+" { pointer-events:none; } #"+n.Ou(t.g!=null?t.g:"mol"+n.Du)+` .event  { pointer-events:all; } line { stroke-linecap:round; } polygon { stroke-linejoin:round; } </style>
`:'<g style="font-size:'+t.k+'px; fill-opacity:1; stroke-opacity:1; fill:black; stroke:black; font-weight:normal; text-rendering:optimizeLegibility; font-family:sans-serif; stroke-linejoin:round; stroke-linecap:round; stroke-dashoffset:0;">',i+="  ",i+=D,t.i){for(P=new U.rrb(t.b);P.a<P.c.a.length;)r=U.qrb(P),n.Iu(t,r);for(e=new U.rrb(t.a);e.a<e.c.a.length;)A=U.qrb(e),n.Iu(t,A);}return t.i||n.Iu(t,"</g>"),i+t.c.a+"</svg>";},n.Iu=function(t,A){h.Ikb(t.c,"  "),h.Ikb(t.c,A),h.Ikb(t.c,d1);},n.Ju=function(t,A,e){n._c(),n.de.call(this,t,A),n.Eu(this),this.g=e,++n.Du;},n.Ou=function(t){var A,e,r,P,i;for(i=new h.Kkb(),P=w.SAb(t).length,r=0;r<P;r++)switch(A=aD(t,r,w.SAb(t).length),e=A&e1,A){case 38:i.a+="&amp;";break;case 60:i.a+="&lt;";break;case 62:i.a+="&gt;";break;case 34:i.a+="&quot;";break;case 39:i.a+="&apos;";break;default:i.a+=String.fromCharCode(e);}return i.a;},x(259,121,{121:1},n.Ju),s.qb=function(t){var A;A='<line x1="'+h.Lib(M.klb(M.glb(new M.llb(t.a),(M.tnb(),M.pnb))))+Y1+'y1="'+h.Lib(M.klb(M.glb(new M.llb(t.c),M.pnb)))+Y1+'x2="'+h.Lib(M.klb(M.glb(new M.llb(t.b),M.pnb)))+Y1+'y2="'+h.Lib(M.klb(M.glb(new M.llb(t.d),M.pnb)))+Y1+yP+this.d+Y1+fr+this.j+Dr,n.Iu(this,A);},s.rb=function(t){var A,e;A=h.Lib(M.klb(M.glb(new M.llb(3*this.j),(M.tnb(),M.pnb)))),e='<line stroke-dasharray="'+A+","+A+Y1+'x1="'+h.Lib(M.klb(M.glb(new M.llb(t.a),M.pnb)))+Y1+'y1="'+h.Lib(M.klb(M.glb(new M.llb(t.c),M.pnb)))+Y1+'x2="'+h.Lib(M.klb(M.glb(new M.llb(t.b),M.pnb)))+Y1+'y2="'+h.Lib(M.klb(M.glb(new M.llb(t.d),M.pnb)))+Y1+yP+this.d+Y1+"stroke-width:"+this.j+Dr,n.Iu(this,e);},s.sb=function(t){var A,e;for(e=new h.Mkb('<polygon points="'),A=0;A<t.a;A++)h.Gkb(e,b1(d.Math.round(t.b[A]))),e.a+=",",h.Gkb(e,b1(d.Math.round(t.c[A]))),e.a+=" ";h.Ikb(e,'" fill="'+this.d+Y1+yP+this.d+Y1+fr+this.j+Dr),n.Iu(this,e.a);},s.tb=function(t,A,e){var r,P,i,D;i=(r=(D=j1.Hgb(this.e,t),new B0._gb(D)).b,r),P='<text x="'+h.Lib(M.klb(M.glb(new M.llb(A-i/2),(M.tnb(),M.pnb))))+Y1+'y="'+h.Lib(M.klb(M.glb(new M.llb(e+this.k/3),M.pnb)))+Y1+'stroke="none" font-size="'+this.e.b+Y1+'fill="'+n.Ou(this.d)+'">'+n.Ou(t)+"</text>",n.Iu(this,P);},s.ub=function(t,A,e){var r;r='<circle cx="'+h.Lib(M.klb(M.glb(new M.llb(t+e/2),(M.tnb(),M.pnb))))+Y1+'cy="'+h.Lib(M.klb(M.glb(new M.llb(A+e/2),M.pnb)))+Y1+'r="'+h.Lib(M.klb(M.glb(new M.llb(e/2),M.pnb)))+Y1+'fill="'+this.d+Dr,n.Iu(this,r);},s.vb=function(t){return n.Fu(this,t);},s.wb=function(){return n.Gu(this);},s.yb=function(t,A,e,r){var P,i;P=n.Ll(this.L,t),i='<circle id="'+n.Ou(this.g!=null?this.g:"mol"+n.Du)+":Atom:"+t+Y1+$8+(P==0?"":'data-atom-map-no="'+P+Y1)+'cx="'+h.Lib(M.klb(M.glb(new M.llb(e),(M.tnb(),M.pnb))))+Y1+'cy="'+h.Lib(M.klb(M.glb(new M.llb(r),M.pnb)))+Y1+'r="8'+Y1+a8,U.sj(this.a,i);},s.zb=function(t,A,e,r,P){var i;i='<line id="'+n.Ou(this.g!=null?this.g:"mol"+n.Du)+":Bond:"+t+Y1+$8+'x1="'+h.Lib(M.klb(M.glb(new M.llb(A),(M.tnb(),M.pnb))))+Y1+'y1="'+h.Lib(M.klb(M.glb(new M.llb(e),M.pnb)))+Y1+'x2="'+h.Lib(M.klb(M.glb(new M.llb(r),M.pnb)))+Y1+'y2="'+h.Lib(M.klb(M.glb(new M.llb(P),M.pnb)))+Y1+fr+8+Y1+a8,U.sj(this.b,i);},s.Ab=function(t){this.j=nn(b1(d.Math.round(100*d.Math.max(t,1))))/100;},s.Bb=function(t){this.d="rgb("+((t&W0)>>16)+","+((t&Y0)>>8)+","+(t&255)+")";},s.Cb=function(t){this.k!=t&&(this.k=t,this.e=new j1.Jgb(t));},s.Db=function(t,A){return this.n=Jt(b1(d.Math.round(t.b))),this.f=Jt(b1(d.Math.round(t.a))),n.$d(this,t,A);},s.pb=function(){return n.Hu(this);},s.f=0,s.i=false,s.j=0,s.k=0,s.n=0,n.Du=0,o.G8=N(259),n.Zu=function(t,A){var e;if(t.c!=null){for(e=0;e<t.c.length;e++)if(A==t.c[e]&&t.d[e])return  true;}return  false;},n.$u=function(t,A,e,r,P){this.a=t,this.b=A,this.g=e,this.f=r,this.e=P;},x(184,1,{184:1},n.$u),s.a=0,s.b=0,s.e=false,s.f=false,s.g=0,o.K8=N(184),n._u=function(t,A){!t.b&&(t.b=new Lt.Q4()),Lt.K4(t.b,h.qjb(A));},n.av=function(t,A,e,r){var P,i,D,u,Q,l;if(P=n.Uk(A,t.d),n.jn(A,P,t.e),n.rn(A,P,t.o,false),n.gn(A,P,t.a),pt(t.c,0)&&(pt(J(t.c,2),0)?(t.c=J(t.c,-3),A.u[P]|=r0):A.u[P]&=ae,n.vn(A,P,t.c,true)),t.b){for(l=B(o.Q7,H,6,t.b.d.a.length,15,1),Q=0;Q<t.b.d.a.length;Q++)l[Q]=Lt.M4(t.b,Q).a;n.pn(A,P,l),w.kAb(t.b.d.a,0);}else if(String.fromCharCode(e).toLowerCase().charCodeAt(0)==e&&h.aib(e)){if(t.d!=5&&t.d!=6&&t.d!=7&&t.d!=8&&t.d!=15&&t.d!=16&&t.d!=33&&t.d!=34)throw st(new h.cC("SmilesParser: atomicNo "+t.d+" must not be aromatic. Position:"+(r-1)));A.u[P]|=r0;}else A.u[P]&=ae;if(t.f)for(u=new U.rrb(t.f);u.a<u.c.a.length;)D=U.qrb(u),t.r=true,n.Cn(D,0,0),n.On(A,true),n.dl(A,D,P);return t.g!=-1&&t.d!=1&&(i=B(o.M7,Y,6,1,15,1),i[0]=t.g<<24>>24,n.nn(A,P,i)),P;},n.bv=function(t,A){var e;for(e=0;A<t.length&&(t[A]!=93||e!=0);)t[A]==91?++e:t[A]==93&&--e,++A;if(A==t.length)throw st(new h.cC("SmilesParser: No matching closing bracket found."));return A+1;},n.cv=function(t,A,e){var r,P;if(A<3||(r=A-1,(t[r]==43||t[r]==45)&&--r,!AA(t[r]&e1))||(--r,t[r]!=68&&t[r]!=82&&t[r]!=88&&t[r]!=122))return  false;for(e[0]=0;r>0&&tu(t[r-1]);)--r,++e[0];for(P=A+1;tu(t[r]);){if(t.length<=P||t[r]!==t[P])return  false;++r,++P;}return  true;},n.dv=function(t,A,e,r,P,i){var D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;if(A[e-1]==36){if(t.r=true,t.q=new U.Mj(),e+=n.hv(t,A,e-1,t.q)-1,A[e++]!=93){if(!i)throw st(new h.cC("SmilesParser: A positive recursive SMARTS followed by another one or by atom query features is not supported. Position:"+(e-1)));(t.k&128)==0&&(e=n.bv(A,e));}return e;}if(A[e-1]==42)t.d=6,t.c=Pt(t.c,1);else if(A[e-1]==63)t.d=0;else if(l=A[e-1]==33,l&&(t.r=true,t.c=Pt(t.c,1),++e),A[e-1]==82&&P&&(AA(A[e]&e1)||t.j&&A[e]==123))t.d=6,t.c=Pt(t.c,1),--e,l&&--e;else {if(L=new n.jv(),!n.ev(t,A,e-1,r,L))throw st(new h.cC("SmilesParser: Unexpected character in atom definition:'"+String.fromCharCode(A[e-1]&e1)+ti+(e-1)));if(t.d=L.a,e+=L.b-1,(t.k&3)!=2&&(t.g=9),P&&(A[e]==44||l)){for(c=L.d,T=L.c,p=e-L.b;p<r;){if(!n.ev(t,A,p,r,L)){if(!l)throw st(new h.cC("SmilesParser: Unexpected character in atom list:'"+String.fromCharCode(A[p]&e1)+Ai+p));break;}if(L.a==1){if(!l)throw st(new h.cC("SmilesParser: Hydrogen is not supported in positive atom lists:'"+h.Zjb((U.wrb(A,p,r),U.Brb(A,B(o.M7,Y,6,r-p,15,1),p,r)))+Ai+p));}else n._u(t,L.a),c=c|L.d,T=T|L.c;if(p+=L.b,l&&A[p]!=59&&A[p]!=38||!l&&A[p]!=44||l&&A[p+1]!=33)break;++p,A[p]==33&&++p;}t.b&&t.b.d.a.length>1&&(t.g=-1,T?c||(t.c=Pt(t.c,4)):t.c=Pt(t.c,2)),e=p;}}for(g=new n.nv(A),v=B(o.Q7,H,6,1,15,1),S=true;S;){if(A[e]==64){++e,A[e]==64&&(t.i=true,++e),t.p=true;continue;}if(A[e]==58){for(++e;AA(A[e]&e1);)t.o=10*t.o+A[e]-48,++e;continue;}if(A[e]==91)throw st(new h.cC("SmilesParser: nested square brackets found. Position:"+e));if(A[e]==93){++e,S=false;continue;}if(t.e=n.gv(A,e,v),v[0]!=0){e+=v[0],t.e==0&&(t.c=Pt(t.c,re));continue;}if(Q=A[e]==33,Q&&++e,A[e]==72){++e,e+=n.kv(g,e,1),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,128)),g.c<=1&&g.b>=1&&(D=Pt(D,256)),g.c<=2&&g.b>=2&&(D=Pt(D,512)),g.c<=3&&g.b>=3&&(D=Pt(D,f0)),Q?(t.c=Pt(t.c,D),t.g=-1):g.b==g.c?t.g=g.c:(t.c=Pt(t.c,J(X0,tA(D))),t.g=-1);continue;}if(A[e]==68||A[e]==100){++e,e+=n.kv(g,e,1),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,r0)),g.c<=1&&g.b>=1&&(D=Pt(D,yA)),g.c<=2&&g.b>=2&&(D=Pt(D,UA)),g.c<=3&&g.b>=3&&(D=Pt(D,TA)),g.c<=4&&g.b>=4&&(D=Pt(D,j0)),nt(D,0)!=0&&(Q?t.c=Pt(t.c,D):pt(J(t.c,aA),0)?t.c=J(t.c,tA(D)):(D=en(D,aA),t.c=Pt(t.c,D)));continue;}if(A[e]==122&&t.j){++e,e+=n.kv(g,e,4),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,Cn)),g.c<=1&&g.b>=1&&(D=Pt(D,ni)),g.c<=2&&g.b>=2&&(D=Pt(D,ei)),g.c<=3&&g.b>=3&&(D=Pt(D,ri)),g.c<=4&&g.b>=4&&(D=Pt(D,fe)),nt(D,0)!=0&&(Q?t.c=Pt(t.c,D):pt(J(t.c,Ln),0)?t.c=J(t.c,tA(D)):(D=en(D,Ln),t.c=Pt(t.c,D)));continue;}if(A[e]==88){if(++e,e+=n.kv(g,e,1),q=(n.Rk(),n.Mk)[t.d],q==null)continue;E=q[0],y=n.gv(A,e,v),v[0]!=0&&(n.qo(t.d)?E+=y:t.d==6?E-=d.Math.abs(y):E-=y),D=0,E-g.c<=0&&E-g.b>=0&&(D=Pt(D,N0)),E-g.c<=1&&E-g.b>=1&&(D=Pt(D,LA)),E-g.c<=2&&E-g.b>=2&&(D=Pt(D,H1)),nt(D,0)!=0&&(Q?t.c=Pt(t.c,D):pt(J(t.c,$A),0)?t.c=J(t.c,tA(D)):(D=en(D,$A),t.c=Pt(t.c,D)));continue;}if(A[e]==65||A[e]==97){t.c=Pt(t.c,Q^A[e]==65?4:2),++e;continue;}if(A[e]==82){++e,e+=n.kv(g,e,3),D=0,g.c<=0&&g.b>=0&&(D=Pt(D,8)),g.c<=1&&g.b>=1&&(D=Pt(D,16)),g.c<=2&&g.b>=2&&(D=Pt(D,32)),g.c<=3&&g.b>=3&&(D=Pt(D,64)),g.b>3&&n.Hv(t.n,(Q?"!R":"R")+g.b),nt(D,0)!=0&&(Q?t.c=Pt(t.c,D):pt(J(t.c,120),0)?t.c=J(t.c,tA(D)):(D=en(D,120),t.c=Pt(t.c,D)));continue;}if(A[e]==114){if(++e,e+=n.kv(g,e,1),g.a){Q?t.c=Pt(t.c,384):t.c=Pt(t.c,8);continue;}k=g.c,g.b>g.c&&n.Hv(t.n,(Q?"!r":"r")+("{"+g.c+"-"+g.b+"}")),!Q&&k>=3&&k<=7?t.c=Pt(t.c,o.Kfb(k<<22)):g.b>g.c||n.Hv(t.n,(Q?"!r":"r")+k);continue;}if(A[e]==118){++e,e+=n.kv(g,e,1),E=g.c,g.b>g.c&&n.Hv(t.n,(Q?"!v":"v")+("{"+g.c+"-"+g.b+"}")),!Q&&E<=14?t.a=E:g.b>g.c||n.Hv(t.n,(Q?"!v":"v")+E);continue;}if(A[e]==94){if(++e,u=A[e++]-48,u<1||u>3)throw st(new h.cC("SmilesParser: Unsupported hybridization. Position:"+e));b=u==1?H1:u==2?LA:N0,Q||(b=J($A,tA(b))),t.c=Pt(t.c,b);continue;}if(A[e]==36){if(!Q)throw st(new h.cC("SmilesParser: non-negated recursive SMARTS relating to preceding atom are not supported yet. Position:"+e));!t.f&&(t.f=new U.Mj()),t.r=true,e+=n.hv(t,A,e,t.f);continue;}if(P&&(A[e]==59||A[e]==38)){t.r=true,++e;continue;}if(P&&A[e]==44&&n.cv(A,e,v)){t.r=true,e+=v[0]+1;continue;}if(P&&A[e]==44&&(t.k&128)!=0){t.r=true,e+=1;break;}throw A[e]==44?st(new h.cC("SmilesParser: alternative atom definitions not supported. (Tip: enumerate SMARTS): '"+String.fromCharCode(A[e]&e1)+TP+e)):st(new h.cC("SmilesParser: unexpected character inside brackets: '"+String.fromCharCode(A[e]&e1)+TP+e));}return e;},n.ev=function(t,A,e,r,P){var i,D;if(P.d=true,P.c=true,A[e]==35){for(++e,t.r=true,P.a=0,P.b=1;e<r&&AA(A[e]&e1);)P.a=10*P.a+A[e]-48,++P.b,++e;if(P.a==0||P.a>=(n.Rk(),n.Lk).length)throw st(new h.cC("SmilesParser: Atomic number out of range. position:"+(e-1)));return  true;}if(A[e]>=65&&A[e]<=90){if(P.b=A[e+1]>=97&&A[e+1]<=122?2:1,P.a=(n.Rk(),n.po(h.xkb(w.qAb(A,e,(D=P.b,q1.Cnb(),D))),321)),P.b==2&&P.a==0&&(P.b=1,P.a=n.po(h.xkb(w.qAb(A,e,(i=P.b,i))),321)),P.d=false,P.a==0)throw st(new h.cC("SmilesParser: Unknown atom label. position:"+(e-1)));return  true;}return A[e]==65&&A[e+1]==115||A[e]==83&&A[e+1]==101?(P.b=2,P.a=(n.Rk(),n.po(h.xkb(w.qAb(A,e,(i=P.b,q1.Cnb(),i))),321)),P.c=false,true):A[e]==99||A[e]==110||A[e]==111||A[e]==112||A[e]==115?(P.b=1,P.a=(n.Rk(),n.po(h.xkb(w.qAb(A,e,(i=P.b,q1.Cnb(),i))),321)),P.c=false,true):false;},n.fv=function(t,A,e,r,P){if(A[e-1]==42)t.d=6,t.c=Pt(t.c,1);else if(A[e-1]==63)t.d=0;else if((A[e-1]==65||A[e-1]==97)&&P)t.d=6,t.c=Pt(t.c,1),t.c=Pt(t.c,A[e-1]==65?4:2),t.r=true;else switch(String.fromCharCode(A[e-1]).toUpperCase().charCodeAt(0)){case 66:e<r&&A[e]==114?(t.d=35,++e):t.d=5;break;case 67:e<r&&A[e]==108?(t.d=17,++e):t.d=6;break;case 70:t.d=9;break;case 73:t.d=53;break;case 78:t.d=7;break;case 79:t.d=8;break;case 80:t.d=15;break;case 83:t.d=16;}return e;},n.gv=function(t,A,e){var r,P;if(e[0]=0,t[A]==43||t[A]==45){for(P=t[A],r=1,++e[0];t[A+e[0]]==P;)++r,++e[0];return r==1&&AA(t[A+1]&e1)&&(r=t[A+1]-48,++e[0]),P==43?r:-r;}return 0;},n.hv=function(t,A,e,r){var P,i,D,u,Q;if(A[e+1]!=40)throw st(new h.cC("SmilesParser: '$' for recursive SMARTS must be followed by '('. position:"+e));for(u=1,i=e+2;i<A.length&&u>0;)A[i]==40?++u:A[i]==41&&--u,++i;if(u>0)throw st(new h.cC("SmilesParser: Missing closing ')' for recursive SMARTS. '('-position:"+(e+1)));if(D=new n.uw(16,16),Q=new n.Jv(t.k),n.Fv(Q,t.n.f),n.xv(Q,D,A,e+2,i-1,true,true),n.On(D,true),w.iAb(r.a,D),A[e-1]==33)for(P=0;P<D.q;P++)n.vn(D,P,i0,true);return i-e;},n.iv=function(t,A){this.n=t,this.k=A,this.j=(A&16)==0,this.d=-1,this.e=0,this.o=0,this.a=-1,this.g=-1,this.c=0;},x(283,1,{},n.iv),s.a=0,s.c=0,s.d=0,s.e=0,s.g=0,s.i=false,s.j=false,s.k=0,s.o=0,s.p=false,s.r=false,o.J8=N(283),n.jv=function(){this.a=-1;},x(370,1,{},n.jv),s.a=0,s.b=0,s.c=false,s.d=false,o.H8=N(370),n.kv=function(t,A,e){var r,P,i,D,u;if(t.a=false,t.d=A,AA(t.e[A]&e1)){for(u=n.lv(t),t.c=t.b=u,r=A-1;r>1&&t2(t.e[r-1]);)--r;for(;t.e[t.d]==44;){for(D=true,i=A-r,P=0;P<i;P++)if(t.e[r+P]!==t.e[t.d+1+P]){D=false;break;}if(!D)break;t.d+=1+i,u=n.lv(t),t.c>u?t.c=u:t.b<u&&(t.b=u);}return t.d-A;}return t.e[A]==123&&AA(t.e[A+1]&e1)?(++t.d,t.c=n.lv(t),t.e[t.d++]!=45||!AA(t.e[t.d]&e1)||(t.b=n.lv(t),t.e[t.d++]!=125)?0:t.d-A):(t.c=1,t.b=e,t.a=true,0);},n.lv=function(t){var A;return A=t.e[t.d++]-48,AA(t.e[t.d]&e1)&&(A=10*A+(t.e[t.d++]-48)),A;},n.mv=function(t){return "{"+t.c+"-"+t.b+"}";},n.nv=function(t){this.e=t;},x(371,1,{},n.nv),s.pb=function(){return n.mv(this);},s.a=false,s.b=0,s.c=0,s.d=0,o.I8=N(371),n.pv=function(t,A,e,r,P,i){var D;return D=n.Wk(t.k,A,e,r),U.rob(t.s,A+"_"+e,j(F(o.Q7,1),H,6,15,[P,i])),U.rob(t.s,e+"_"+A,j(F(o.Q7,1),H,6,15,[P,i])),U.qsb(t.a,A),U.qsb(t.a,e),D;},n.qv=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b;for(l=B(o.Q7,H,6,t.k.f,15,1),u=B(o.Q7,H,6,t.k.f,15,1),Q=B(o.Q7,H,6,t.k.f,15,1),L=B(o.Q7,H,6,t.k.f,15,1),e=n.Zl(t.k,0,A),r=n.Zl(t.k,1,A),u[0]=e,u[1]=r,Q[0]=-1,Q[1]=A,l[e]=1,l[r]=2,L[e]=-1,L[r]=e,D=1,y=1;D<=y&&l[u[D]]<15;){for(b=u[D],T=0;T<n.$o(t.k,b);T++)if(P=n.Zo(t.k,b,T),P!=L[b]){if(i=n._o(t.k,b,T),P==e){for(Q[0]=i,c=0;c<=y;c++)t.g[Q[T]]||(t.g[Q[T]]=true,++t.c);return;}n.Km(t.k,P)&&l[P]==0&&(++y,u[y]=P,Q[y]=i,l[P]=l[b]+1,L[P]=b);}++D;}},n.rv=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c;for(n.bw(t.k,7),y=false,T=B(o.Q7,H,6,2,15,1),c=B(o.Q7,H,6,2,15,1),L=B(o.Q7,H,6,2,15,1),r=0;r<t.k.g;r++)if(!n.jq(t.k,r)&&n.im(t.k,r)==2){for(D=0;D<2;D++){for(T[D]=-1,L[D]=-1,A=n.Zl(t.k,D,r),l=0;l<n.$o(t.k,A);l++)P=n._o(t.k,A,l),P!=r&&(T[D]==-1&&(n.im(t.k,P)==257||n.im(t.k,P)==129)?(T[D]=n.Zo(t.k,A,l),c[D]=P):L[D]=n.Zo(t.k,A,l));if(T[D]==-1)break;}if(T[0]!=-1&&T[1]!=-1){for(Q=n.im(t.k,c[0])==n.im(t.k,c[1]),u=0;u<2;u++)T[u]==n.Zl(t.k,0,c[u])&&(Q=!Q);for(i=0;i<2;i++)L[i]!=-1&&L[i]<T[i]&&(Q=!Q);n.Jn(t.k,r,Q?2:1,false),y=true;}}for(e=0;e<t.k.g;e++)(n.im(t.k,e)==257||n.im(t.k,e)==129)&&n.Mn(t.k,e,1);return y;},n.sv=function(t){return t==61?2:t==35?4:t==36?32:t==58?8:t==62?16:t==126?127:1;},n.tv=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;for(e=0;e<t.k.f;e++)if(n.Km(t.k,e))for(Q=B(o.Q7,H,6,t.k.f,15,1),u=B(o.Q7,H,6,t.k.f,15,1),l=B(o.Q7,H,6,t.k.f,15,1),u[0]=e,Q[e]=1,l[e]=-1,D=0,L=0;D<=L;){for(P=(Q[u[D]]&1)==1?1:2,y=0;y<n.$o(t.k,u[D]);y++)if(r=n._o(t.k,u[D],y),n.fm(t.k,r)==P&&A[r]&&(i=n.Zo(t.k,u[D],y),Q[i]==0)){if(P==1&&n.Km(t.k,i)){for(T=u[D];T!=-1;)n.Mn(t.k,n.Xo(t.k,i,T),P==1?2:1),P=3-P,i=T,T=l[T];return n.sn(t.k,e,false),n.sn(t.k,i,false),t.b-=2,true;}u[++L]=i,l[i]=u[D],Q[i]=Q[u[D]]+1;}++D;}return  false;},n.uv=function(t){var A,e,r,P;for(A=0;A<t.k.f;A++)if(n.Tl(t.k,A)==7&&n.Bl(t.k,A)==0&&n.vp(t.k,A)>3&&n.Qo(t.k,A)>0){for(P=0;P<n.$o(t.k,A);P++)if(e=n.Zo(t.k,A,P),r=n._o(t.k,A,P),n.fm(t.k,r)>1&&n.Hm(t.k,e)){n.im(t.k,r)==4?n.Mn(t.k,r,2):n.Mn(t.k,r,1),n.jn(t.k,A,n.Bl(t.k,A)+1),n.jn(t.k,e,n.Bl(t.k,e)-1),n.gn(t.k,A,-1);break;}}},n.vv=function(t){return t.r?"Unresolved SMARTS features:"+t.r:"";},n.wv=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt;for(n.bw(t.k,1),t.g=B(o.vfb,Tt,6,t.k.g,16,1),t.c=0,y=0;y<t.k.g;y++)n.im(t.k,y)==8&&(n.Mn(t.k,y,1),t.g[y]=true,++t.c);for(p=B(o.vfb,Tt,6,t.k.f,16,1),xt=new n.Dt(t.k,3),S=B(o.vfb,Tt,6,xt.j.a.length,16,1),It=0;It<xt.j.a.length;It++){for(Ht=U.xj(xt.j,It),S[It]=true,G=Ht,tt=0,it=G.length;tt<it;++tt)if(q=G[tt],!n.Km(t.k,q)){S[It]=false;break;}if(S[It]){for(K=Ht,Qt=0,lt=K.length;Qt<lt;++Qt)q=K[Qt],p[q]=true;for(Kt=U.xj(xt.k,It),V=Kt,O=0,z=V.length;O<z;++O)q=V[O],t.g[q]||(t.g[q]=true,++t.c);}}for(T=0;T<t.k.g;T++)!t.g[T]&&(xt.c[T]&e1)!=0&&n.Km(t.k,n.Zl(t.k,0,T))&&n.Km(t.k,n.Zl(t.k,1,T))&&n.qv(t,T);for(c=0;c<t.k.g;c++)t.g[c]||(i=n.Zl(t.k,0,c),D=n.Zl(t.k,1,c),!p[i]&&!p[D]&&n.Km(t.k,i)&&n.Km(t.k,D)&&(t.g[c]=true,++t.c));for(n.bw(t.k,7),(t.q==2||t.q==1&&t.p)&&n.Dv(t),v=B(o.vfb,Tt,6,t.k.g,16,1),t.k.g>=0&&h.Qkb(t.g,0,v,0,t.k.g),Et=0;Et<xt.j.a.length;Et++)if(S[Et]){for(Ht=U.xj(xt.j,Et),Dt=Ht,ot=0,bt=Dt.length;ot<bt;++ot)if(ut=Dt[ot],!n.Ev(t,ut))for(n.Km(t.k,ut)&&(n.sn(t.k,ut,false),--t.b),q=0;q<n.$o(t.k,ut);q++)g=n._o(t.k,ut,q),t.g[g]&&(t.g[g]=false,--t.c);}for(n.Cv(t),ct=0;ct<xt.j.a.length;ct++)if(S[ct]&&U.xj(xt.k,ct).length==6){for(Kt=U.xj(xt.k,ct),E=true,Q=Kt,l=0,L=Q.length;l<L;++l)if(u=Q[l],!t.g[u]){E=false;break;}E&&(n.Bv(t,Kt[0]),n.Bv(t,Kt[2]),n.Bv(t,Kt[4]),n.Cv(t));}for(wt=5;wt>=4;wt--)do for(Ut=false,u=0;u<t.k.g;u++)if(t.g[u]){for(r=0,k=0;k<2;k++)for(b=n.Zl(t.k,k,u),q=0;q<n.$o(t.k,b);q++)t.g[n._o(t.k,b,q)]&&++r;if(r==wt){n.Bv(t,u),n.Cv(t),Ut=true;break;}}while(Ut);for(;t.b>=2&&n.tv(t,v););if(A){if(t.b!=0)for(P=0;P<t.k.f;P++)n.Km(t.k,P)&&(n.sn(t.k,P,false),n.vn(t.k,P,2,true),--t.b,e=true);if(t.c!=0)for(u=0;u<t.k.g;u++)t.g[u]&&(t.g[u]=false,n.Mn(t.k,u,8),--t.c,e=true);}else for(P=0;P<t.k.f;P++)n.Km(t.k,P)&&n.op(t.k,P)!=0&&(n.sn(t.k,P,false),n.wn(t.k,P,32),--t.b);if(t.q==0||t.q==1&&!e){if(t.b!=0)throw st(new h.cC(Pi));if(t.c!=0)throw st(new h.cC(Pi));}},n.xv=function(t,A,e,r,P,i,D){var u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt,c1,D1,St,gt,w1,I1,L1,G1,s1,Q1;for(t.k=A,n._v(t.k),t.s=new U.pub(),t.a=new U.rsb(),t.r&&h.whb(t.r,0),t.b=0,t.p=false,l=t.q!=0,Nt=null,b=B(o.Q7,H,6,32,15,1),b[0]=-1,kt=B(o.Q7,H,6,16,15,1),D1=B(o.Q7,H,6,16,15,1),c1=B(o.Q7,H,6,16,15,1),Bt=B(o.Q7,H,6,16,15,1),ut=0;ut<16;ut++)kt[ut]=-1;for(T=0,O=-1,St=false,Ut=false,z=false,p=0,v=1,k=0;e[r]<=32;)++r;for(;r<P;){if(gt=e[r++]&e1,h.aib(gt)||gt==42||gt==63||gt==33&&l&&St||gt==35&&l&&St||gt==36&&l&&St){if(c=new n.iv(t,t.j|t.q),St){if((t.j&128)!=0){if(V=new n.Lv(r-1),r=n.dv(c,e,r,P,true,true),e[r-1]!=93){for(;e[r-1]!=93;)r=n.dv(c,e,r+1,P,true,true),++V.a;t.f.add(V);}}else r=n.dv(c,e,r,P,l,false);}else r=n.fv(c,e,r,P,l);if(St=false,c.q&&U.xj(c.q,0)){O=b[p],b[p]=A.q,n.Yk(A,c.q?U.xj(c.q,0):null),O!=-1&&v!=512&&(g=n.Wk(t.k,O,O,v),k!=0&&(t.p=true,n.Ln(t.k,g,k,true),n.Sk(t.k,g))),v=1,k=0;continue;}if(c.d==-1&&gt!=63)throw st(new h.cC("SmilesParser: unknown element label found. Position:"+(r-1)));c.r&&(t.p=true),y=n.av(c,t.k,gt,r),n.Km(t.k,y)&&++t.b,O=b[p],O!=-1&&v!=512&&(g=n.Wk(t.k,O,y,v),k!=0&&(t.p=true,n.Ln(t.k,g,k,true),n.Sk(t.k,g))),v=1,k=0,b[p]=y,T!=0&&(n.tn(t.k,y,T),T=0),D&&(Kt=Nt?U.eqb(Nt,h.qjb(O)):null,Kt&&n.Nv(Kt,y,(wt=r,c.d==1&&n.Ml(t.k,y)==0,wt)),c.p&&(!Nt&&(Nt=new U.Xwb()),it=c.g==9?0:c.g,U.Pwb(Nt,h.qjb(y),new n.Tv(t,y,r-2,O,it,c.i))));continue;}if(gt==46){b[p]=-1,v=512;continue;}if(gt==45||gt==61||gt==35||gt==36||gt==58||gt==47||gt==92||gt==60||gt==126||gt==33||gt==64){if(St)throw st(new h.cC("SmilesParser: unexpected bond symbol inside square brackets: '"+String.fromCharCode(gt)+TP+(r-1)));for(G=0;gt==45||gt==61||gt==35||gt==36||gt==58||gt==47||gt==92||gt==60||gt==126||gt==33||gt==64;){if(gt==33){if(gt=e[r++]&e1,gt==64)k|=128;else if(gt==45&&e[r]==62||gt==60&&e[r]==45)G|=16,++r;else if(gt==45)G|=1;else if(gt==61)G|=2;else if(gt==35)G|=4;else if(gt==36)G|=32;else if(gt==58)G|=8;else throw st(new h.cC("SmilesParser: bond symbol '"+String.fromCharCode(gt)+"' not allowed after '!'. Position:"+(r-1)));}else if(gt==64?k|=256:gt==61?v=2:gt==35?v=4:gt==36?v=32:gt==58?v=8:gt==126?k|=15:gt==47?D&&(v=257):gt==92?D&&(v=129):(gt==45&&e[r]==62||gt==60&&e[r]==45)&&(v=16,++r),e[r]==44)for(k|=(v==16?62:gt)==61?2:(v==16?62:gt)==35?4:(v==16?62:gt)==36?32:(v==16?62:gt)==58?8:(v==16?62:gt)==62?16:(v==16?62:gt)==126?127:1;e[r]==44;)e[r+1]==60&&e[r+2]==45||e[r+1]==45&&e[r+2]==62?(k|=16,r+=3):(k|=n.sv(e[r+1]&e1),r+=2);if(e[r]==59){++r,gt=e[r++]&e1;continue;}G!=0&&(k|=127&~G);break;}continue;}if(gt<=32){r=P;continue;}if(AA(gt)){if(It=gt-48,St){for(;r<P&&AA(e[r]&e1);)It=10*It+e[r]-48,++r;T=It;}else {if(S=Ut?r-3:r-2,Qt=e[S]==45||e[S]==47||e[S]==92||e[S]==61||e[S]==35||e[S]==36||e[S]==58||e[S]==62||e[S]==126,Ut&&r<P&&AA(e[r]&e1)&&(It=10*It+e[r]-48,Ut=false,++r),It>=kt.length){if(It>=100)throw st(new h.cC("SmilesParser: ringClosureAtom number out of range: "+It));for(Et=kt.length,ct=kt.length;ct<=It;)ct=d.Math.min(100,ct+16);for(kt=(w.CAb(ct),U.Brb(kt,B(o.Q7,H,6,ct,15,1),0,ct)),D1=(w.CAb(ct),U.Brb(D1,B(o.Q7,H,6,ct,15,1),0,ct)),c1=(w.CAb(ct),U.Brb(c1,B(o.Q7,H,6,ct,15,1),0,ct)),Bt=(w.CAb(ct),U.Brb(Bt,B(o.Q7,H,6,ct,15,1),0,ct)),Dt=Et;Dt<ct;Dt++)kt[Dt]=-1;}if(kt[It]==-1)kt[It]=b[p],D1[It]=r-1,c1[It]=Qt?v:-1,Bt[It]=Qt?k:0;else {if(kt[It]==b[p])throw st(new h.cC("SmilesParser: ring closure to same atom"));D&&Nt&&(Kt=U.eqb(Nt,h.qjb(kt[It])),Kt&&n.Nv(Kt,b[p],D1[It]),Kt=U.eqb(Nt,h.qjb(b[p])),Kt&&n.Nv(Kt,kt[It],r-1)),c1[It]!=-1?v=c1[It]:v==257?v=129:v==129&&(v=257),u=kt[It],Q=b[p],g=n.pv(t,u,Q,v,D1[It],r-1),Bt[It]!=0&&(k=Bt[It]),k!=0&&(t.p=true,n.Ln(t.k,g,Bt[It],true),n.Sk(t.k,g)),kt[It]=-1;}v=1,k=0;}continue;}if(gt==43)throw st(new h.cC("SmilesParser: '+' found outside brackets. Position:"+(r-1)));if(gt==40){if(b[p]==-1){z=true;continue;}++p,b.length==p&&(b=U.zrb(b,b.length+32)),b[p]=b[p-1];continue;}if(gt==41){if(p==0){if(!z)throw st(new h.cC("SmilesParser: Closing ')' without opening counterpart. Position:"+(r-1)));b[0]=-1,z=false;continue;}--p;continue;}if(gt==91){St=true;continue;}if(gt==93)throw st(new h.cC("SmilesParser: closing bracket at unexpected position:"+(r-1)));if(gt==37){Ut=true;continue;}throw st(new h.cC("SmilesParser: unexpected character outside brackets: '"+String.fromCharCode(gt)+ti+(r-1)));}if(v!=1)throw st(new h.cC("SmilesParser: dangling open bond; position:"+r));for(ot=0;ot<kt.length;ot++)if(kt[ot]!=-1)throw st(new h.cC("SmilesParser: dangling ring closure: "+ot+"; position:"+r));for(tt=n.lp(t.k),n.Qn(t.k,true),n.bw(t.k,1),L=0;L<t.k.q;L++)if(n.El(t.k,L)!=null){if(K=n.Fl(t.k,L)[0],t.p||t.q==2){if(K!=9)if(t.i)for(lt=0;lt<K;lt++)n.Wk(t.k,L,n.Uk(t.k,1),1);else K==0&&n.vn(t.k,L,1792,true),K==1&&n.vn(t.k,L,1664,true),K==2&&n.vn(t.k,L,1408,true),K==3&&n.vn(t.k,L,896,true);}else {if(K==9&&(K=0),!n.Lm(t.k,L)&&(!n.Km(t.k,L)||n.Tl(t.k,L)==6&&n.Bl(t.k,L)==0)){for(Q1=n.mo(n.Tl(t.k,L)),E=false,w1=n.vp(t.k,L),w1-=n.mm(t.k,L,w1),w1+=K,n.Km(t.k,L)&&++w1,L1=Q1,G1=0,s1=L1.length;G1<s1;++G1)if(I1=L1[G1],w1<=I1){E=true,I1==w1+2?n.wn(t.k,L,48):I1==w1+1?n.wn(t.k,L,32):(I1!=w1||I1!=Q1[0])&&n.gn(t.k,L,w1);break;}E||n.gn(t.k,L,w1);}if(t.i||!n.Aq(t.k,L))for(lt=0;lt<K;lt++)n.Wk(t.k,L,n.Uk(t.k,1),1);}}else !t.i&&(t.p||t.q==2)&&(K=n.cp(t.k,L),K>=1&&n.vn(t.k,L,128,true),K>=2&&n.vn(t.k,L,256,true),K>=3&&n.vn(t.k,L,512,true),K>=4&&n.vn(t.k,L,f0,true));if(!t.i&&(t.p||t.q==2)&&n.qq(t.k,false),n.bw(t.k,1),n.uv(t),n.wv(t,l,t.p),t.k.t=null,n.Qn(t.k,false),D&&(n.rv(t),Nt)){for(xt=(Ht=new U.Epb(Nt).a.cd().Tb(),new U.Lpb(Ht));xt.a.ed();)Kt=(q=xt.a.fd(),q.jd()),n.un(t.k,tt[Kt.a],n.Ov(Kt,tt),false);n.vq(t.k,0);}n.vq(t.k,0),i&&(bt=new X.CA(t.d),pt(t.n,0)&&X.yA(bt,t.n),X.jA(bt,t.k),D&&n.ow(t.k)),(t.p||t.q==2)&&(n.On(t.k,true),n.Cq(t.k),n.Dq(t.k));},n.yv=function(t,A,e,r,P){n.xv(t,A,e,0,e.length,r,P);},n.zv=function(t,A){var e;return A==null?null:n.Av(t,w.sAb((e=A,q1.Cnb(),e)));},n.Av=function(t,A){var e,r,P,i,D,u,Q,l,L,y;for(P=Lt.n4(A,0);P>0&&A[P-1]==45;)P=Lt.n4(A,P+1);for(i=P==-1?-1:Lt.n4(A,P+1);i>0&&A[i-1]==45;)i=Lt.n4(A,i+1);if(i==-1)throw st(new h.cC("Missing one or both separators ('>')."));if(Lt.n4(A,i+1)!=-1)throw st(new h.cC("Found more than 2 separators ('>')."));for(y=new At.zH(),L=0,r=0;r<A.length;){for(;r<A.length&&A[r]==46;)++r;if(r==A.length)break;for(l=r,Q=-1,D=0,e=r;e<A.length;e++){if(A[e]==40){++D;continue;}if(A[e]==41){if(--D<0)throw st(new h.cC("Found closing bracket without matching opening one."));continue;}if(D==0&&(A[e]==46&&(t.o||e+1<A.length&&A[e+1]==46)||A[e]==62&&(e==0||A[e-1]!=45))){Q=e;break;}}if(Q==-1&&(Q=A.length),l<Q){if(u=new n.tw(),n.xv(t,u,A,l,Q,true,true),t.q==1&&t.p)return n.Av(new n.Jv(t.j|2),A);L==0?(U.sj(y.g,u),y.d=-1):L==1?U.sj(y.a,u):(U.sj(y.f,u),y.d=-1);}for(r=Q;r<A.length&&A[r]==62;)++r,++L;}return y;},n.Bv=function(t,A){var e,r,P,i;for(n.im(t.k,A)==1&&n.Mn(t.k,A,2),P=0;P<2;P++)for(e=n.Zl(t.k,P,A),n.Km(t.k,e)&&(n.sn(t.k,e,false),--t.b),i=0;i<n.$o(t.k,e);i++)r=n._o(t.k,e,i),t.g[r]&&(t.g[r]=false,--t.c);},n.Cv=function(t){var A,e,r,P,i,D,u;do for(u=false,e=0;e<t.k.g;e++)if(t.g[e]){for(i=false,P=0;P<2;P++){for(A=false,r=n.Zl(t.k,P,e),D=0;D<n.$o(t.k,r);D++)if(e!=n._o(t.k,r,D)&&t.g[n._o(t.k,r,D)]){A=true;break;}if(!A){i=true;break;}}i&&(u=true,n.Bv(t,e));}while(u);},n.Dv=function(t){var A,e,r,P,i,D;for(A=0;A<t.k.g;A++)if(t.g[A]){for(i=true,P=0;P<2&&i;P++)for(e=n.Zl(t.k,P,A),D=0;D<n.$o(t.k,e)&&i;D++)A!=n._o(t.k,e,D)&&t.g[n._o(t.k,e,D)]&&(i=false);if(i)for(n.Mn(t.k,A,8),--t.c,r=0;r<2;r++)e=n.Zl(t.k,r,A),n.Km(t.k,e)&&(n.sn(t.k,e,false),--t.b);}},n.Ev=function(t,A){var e,r;return !n.Ft(n.Tl(t.k,A))||n.Tl(t.k,A)==6&&(!n.Km(t.k,A)||n.Bl(t.k,A)>0)||(e=n.El(t.k,A)==null||n.Fl(t.k,A)[0]==9?0:n.Fl(t.k,A)[0],r=n.jp(t.k,A)-e,r<1)?false:n.Tl(t.k,A)==16||n.Tl(t.k,A)==34||n.Tl(t.k,A)==52?n.$o(t.k,A)==2&&n.Bl(t.k,A)<=0?false:r!=2:true;},n.Fv=function(t,A){t.f=A;},n.Gv=function(t,A){t.n=A;},n.Hv=function(t,A){t.e&&(!t.r&&(t.r=new h.Kkb()),h.Ikb(t.r," "),h.Ikb(t.r,A));},n.Iv=function(){n.Jv.call(this,0);},n.Jv=function(t){this.j=t&-4,this.q=t&3,this.o=(t&32)!=0,this.e=(t&64)!=0,this.i=(t&8)!=0,this.d=2,(t&4)!=0&&(this.d|=1),this.i&&(this.d&=-3);},x(113,1,{},n.Iv,n.Jv),s.b=0,s.c=0,s.d=0,s.e=false,s.i=false,s.j=0,s.n=0,s.o=false,s.p=false,s.q=0,o.O8=N(113),n.Kv=function(t,A){return P8(A.b,t.b);},n.Lv=function(t){this.b=t,this.a=1;},x(216,1,{216:1,32:1},n.Lv),s.Sb=function(t){return n.Kv(this,t);},s.a=0,s.b=0,o.L8=N(216),n.Nv=function(t,A,e){if(!t.c){if(t.e.a.length==4){t.c=true;return;}U.sj(t.e,new n.Uv(t,A,e));}},n.Ov=function(t,A){var e,r,P;if(t.c)return 3;for(P=new U.rrb(t.e);P.a<P.c.a.length;)r=U.qrb(P),r.a!=se&&r.a!=qt&&(r.a=A[r.a]);switch(e=false,t.e.a.length){case 2:if(U.xj(t.e,0).a>=se||U.xj(t.e,1).a>=se)return 3;e=n.Rv(t,A);break;case 3:U.sj(t.e,new n.Uv(t,qt,t.b));case 4:e=n.Sv(t);break;default:return 3;}return t.d^e?1:2;},n.Pv=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it;if(p=t.f.k,P=U.xj(t.e,e),it=P.a,y=A[t.a],k=n.op(p,it)>0&&p.j[it]==p.e[it],g=U.nsb(t.f.a,t.f.t[it]),!k&&!g)return D=p.i[it][2],p.C[D]==1&&(D=t.f.t[D],(e==0?D>it:t.f.t[p.i[it][1]]>D)&&(r=!r)),r;if(k&&(i=p.i[it][e],D=t.f.t[i],D>it&&(r=!r,p.C[i]==1&&(r=!r))),g){for(E=p.e[it],c=B(o.Q7,H,6,E-1,15,1),b=B(o.Q7,H,6,E-1,15,1),q=0,v=0,G=0;v<E;v++)D=p.i[it][v],D!=y&&(c[G]=D,l=n.Xo(p,it,D),tt=U.oob(t.f.s,t.f.t[it]+"_"+t.f.t[D]),b[G++]=tt==null?-1:l,tt!=null&&++q);if(k)c[0]<it&&(r=!r);else switch(q){case 1:if(n.op(p,it)>0)break;S=b[0]>=0,T=c[S?0:1],V=c[S?1:0],V>it?T>it&&p.C[V]!=1&&(r=!r):T<it&&(r=!r);break;case 2:u=n.Qv(t,b[0],it),Q=n.Qv(t,b[1],it),Qt=U.oob(t.f.s,it+"_"+t.f.t[u]),z=U.oob(t.f.s,it+"_"+t.f.t[Q]),L=t.b,e==0?(K=Qt[1]<L?Qt[1]:Qt[0],O=z[1]<L?z[1]:z[0]):(K=Qt[0]<L?Qt[1]:Qt[0],O=z[0]<L?z[1]:z[0]),K<O!=u<Q&&(r=!r);}}return r;},n.Qv=function(t,A,e){return t.f.k.D[0][A]==e?t.f.k.D[1][A]:t.f.k.D[0][A];},n.Rv=function(t,A){var e,r;if(r=false,t.f.t==null)for(t.f.t=B(o.Q7,H,6,A.length,15,1),e=t.f.t.length;--e>=0;)t.f.t[A[e]]=e;return r=n.Pv(t,A,0,r),r=n.Pv(t,A,1,r),r;},n.Sv=function(t){var A,e,r;for(e=false,A=1;A<t.e.a.length;A++)for(r=0;r<A;r++)U.xj(t.e,r).a>U.xj(t.e,A).a&&(e=!e),U.xj(t.e,r).b>U.xj(t.e,A).b&&(e=!e);return e;},n.Tv=function(t,A,e,r,P,i){this.f=t,P!=0&&P!=1?this.c=true:(this.a=A,this.b=e,this.d=i,this.e=new U.Mj(),r!=-1&&n.Nv(this,r,e-1),r!=-1&&P==1&&n.Nv(this,se,e+1));},x(217,1,{217:1},n.Tv),s.a=0,s.b=0,s.c=false,s.d=false,o.N8=N(217),n.Uv=function(t,A,e){this.c=t,this.a=A,this.b=e;},x(182,1,{182:1},n.Uv),s.pb=function(){return "["+(this.a==se?"h":this.a==qt?"lp":n.Il(this.c.f.k,this.a))+this.b+"]";},s.a=0,s.b=0,o.M8=N(182),n.Wv=function(t){t.c=new U.Mj();},n.Xv=function(t,A){var e,r,P,i;if(i=t.c.a.length,i==0)return U.rj(t.c,0,A),0;for(P=1;2*P<=i;)P<<=1;for(r=P,--P;r!=0;){if(r>>=1,P>=i){P-=r;continue;}if(e=h.Wjb(A,U.xj(t.c,P)),e==0)return  -1;if(r==0)break;e<0?P-=r:P+=r;}return P<i&&h.Wjb(A,U.xj(t.c,P))>0&&++P,U.rj(t.c,P,A),P;},n.Yv=function(t,A){var e,r,P,i;if(i=t.c.a.length,i==0)return  -1;for(P=1;2*P<=i;)P<<=1;for(r=P,--P;r!=0;){if(r>>=1,P>=i){P-=r;continue;}if(e=h.Wjb(A,U.xj(t.c,P)),e==0)return P;if(r==0)break;e<0?P-=r:P+=r;}return  -1;},n.Zv=function(){n.Wv(this);},x(102,1,ii,n.Zv),s.bc=function(t){return n.Yv(this,t);},o.P8=N(102),n._v=function(t){n.jl(t),t.d=null;},n.aw=function(t,A){n.pl(t,A),t.d&&(A.T=0);},n.bw=function(t,A){var e,r,P,i;if(n.Eo(t,A),(A&~t.T)!=0){for((t.T&15)!=0&&t.q>1&&n.ei(t.J[0],t.J[1])&&X.jA(new X.CA(0),t),t.c&&(A|=128),e=0;e<t.q;e++)t.u[e]&=-67223560;for(r=0;r<t.g;r++)t.F[r]&=-64;P=0,i=0,(A&32)!=0?(P=32,i=1):(A&64)!=0&&(P=64,i=3),(A&128)!=0&&(P|=128,i|=32),t.d=new n.Og(t,i),n.Jg(t.d),n.Kg(t.d),n.Ig(t.d),n.sw(t)&&(t.d=new n.Og(t,i)),t.T|=24|P;}},n.cw=function(t,A){return n.wg(t.d,A);},n.dw=function(t,A){return n.lg(t.d,A);},n.ew=function(t){var A;switch(n.bw(t,31),A=t.I&e1,t.I&fA){case H1:return null;case r0:return A==1?"meso":""+A+" meso diastereomers";case 0:return "unknown chirality";case Ye:return "both enantiomers";case yA:return "this enantiomer";case 327680:return "this or other enantiomer";case tn:return "two epimers";default:return A==1?"one stereo isomer":""+A+" stereo isomers";}},n.fw=function(t){var A;return A=new n.uw(t.q,t.r),n.ol(t,A),A;},n.gw=function(t){var A,e,r,P,i,D,u;for(n.bw(t,15),D=B(o.Q7,H,6,3,15,1),e=0;e<t.f;e++)(t.u[e]&q0)!=0&&(u=(t.u[e]&l1)>>18,u!=0&&(D[u]=d.Math.max(D[u],(t.u[e]&l1)>>18!=1&&(t.u[e]&l1)>>18!=2?-1:(t.u[e]&F0)>>20)));for(P=0;P<t.g;P++)((t.F[P]&3)==1||(t.F[P]&3)==2)&&t.H[P]==1&&(u=(t.F[P]&768)>>8,u!=0&&(D[u]=d.Math.max(D[u],(t.F[P]&768)>>8!=1&&(t.F[P]&768)>>8!=2?-1:(t.F[P]&E0)>>10)));for(i=B(o.Q7,E1,7,3,0,2),i[1]=B(o.Q7,H,6,1+D[1],15,1),i[2]=B(o.Q7,H,6,1+D[2],15,1),A=0;A<t.f;A++)(t.u[A]&q0)!=0&&(u=(t.u[A]&l1)>>18,u!=0&&++i[u][(t.u[A]&l1)>>18!=1&&(t.u[A]&l1)>>18!=2?-1:(t.u[A]&F0)>>20]);for(r=0;r<t.g;r++)((t.F[r]&3)==1||(t.F[r]&3)==2)&&t.H[r]==1&&(u=(t.F[r]&768)>>8,u!=0&&++i[u][(t.F[r]&768)>>8!=1&&(t.F[r]&768)>>8!=2?-1:(t.F[r]&E0)>>10]);return i;},n.hw=function(t){var A,e;return A=B(o.Q7,H,6,t.q,15,1),e=n.hp(t,A,false,false),n.iw(t,A,e);},n.iw=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;for(k=B(o.Q8,GA,26,e,0,1),D=B(o.Q7,H,6,e,15,1),l=B(o.Q7,H,6,e,15,1),i=B(o.Q7,H,6,t.q,15,1),P=0;P<t.q;P++)A[P]!=-1&&(i[P]=D[A[P]]++);for(Q=0;Q<t.r;Q++)b=A[t.D[0][Q]],g=A[t.D[1][Q]],b==g&&b!=-1&&++l[b];for(v=0;v<e;v++)k[v]=new n.uw(D[v],l[v]),n.aw(t,k[v]);for(r=0;r<t.q;r++)A[r]!=-1&&n.ll(t,k[A[r]],r,0,0);for(u=0;u<t.r;u++)b=A[t.D[0][u]],g=A[t.D[1][u]],b==g&&b!=-1&&n.nl(t,k[b],u,0,0,i,false);for(y=k,T=0,c=y.length;T<c;++T)L=y[T],n.cn(L,1),n.cn(L,2);return k;},n.jw=function(t){return n.bw(t,15),!t.d&&(t.f<2||!n.ei(t.J[0],t.J[1]))&&(t.d=new n.Ng(t)),t.d?n.rg(t.d):null;},n.kw=function(t){return n.bw(t,15),!t.d&&(t.f<2||!n.ei(t.J[0],t.J[1]))&&(t.d=new n.Ng(t)),t.d?n.mg(t.d):null;},n.lw=function(t){var A,e;for(n.bw(t,31),e=0,A=0;A<t.f;A++)(t.u[A]&3)!=0&&(t.u[A]&4)==0&&++e;return e;},n.mw=function(t,A){return n.vg(t.d,A);},n.nw=function(t,A){t.c=A,t.T&=-144;},n.ow=function(t){n.bw(t,31),t.d&&n.Lg(t.d);},n.pw=function(t){var A,e;for(n.bw(t,15),t.L=false,A=0;A<t.q;A++)t.u[A]&=Ar,(t.u[A]&3)!=0?t.u[A]|=oe:t.u[A]&=tr;for(e=0;e<t.r;e++)(t.F[e]&3)!=0&&n.fm(t,e)==2?t.H[e]=386:t.H[e]&=-385;t.T&=-249;},n.qw=function(t,A,e){var r,P,i,D,u,Q,l,L,y;if(y=t.u[A]&3,y==1||y==2){if(u=false,t.o[A]==2&&t.j[A]==2&&t.n[A][0]==2&&t.n[A][1]==2&&t.C[A]<=7)for(D=0;D<t.j[A];D++){for(r=t.i[A][D],L=0,l=B(o.Q7,H,6,3,15,1),Q=0;Q<t.j[r];Q++)l[L]=t.i[r][Q],l[L]!=A&&++L;L==2&&l[0]<l[1]^e[l[0]]<e[l[1]]&&(u=!u);}else for(D=1;D<t.j[A];D++)for(Q=0;Q<D;Q++)P=t.i[A][D],i=t.i[A][Q],P<i^e[P]<e[i]&&(u=!u);u&&(y=y==1?2:1);}return y;},n.rw=function(t){var A,e,r,P,i,D;for(n.Bq(t),n.bw(t,31),r=0;r<t.f;r++){if(((t.u[r]&l1)>>18==1||(t.u[r]&l1)>>18==2)&&((t.u[r]&q0)==0||(t.u[r]&3)==3))throw st(new h.cC(hP));if((t.u[r]&H1)!=0)throw st(new h.cC(cP));if(((t.u[r]&3)==1||(t.u[r]&3)==2)&&t.o[r]==0){for(A=B(o.O7,Ft,6,t.j[r],15,1),i=0;i<t.j[r];i++)A[i]=n.Yl(t,r,t.i[r][i]);for(P=1;P<t.j[r];P++)if(!n.Sm(t,t.k[r][P],r)){for(D=0;D<P;D++)if(!n.Sm(t,t.k[r][D],r)&&(e=d.Math.abs(n.oo(A[P],A[D])),e<.08||e>J8))throw st(new h.cC(bP));}}}},n.sw=function(t){var A,e,r,P,i,D,u;for(u=false,e=0;e<t.f;e++)((t.u[e]&q0)==0||(t.u[e]&3)==3)&&(t.u[e]&=Ar);for(i=0;i<t.g;i++)(n.fm(t,i)!=1||(t.F[i]&3)==0||(t.F[i]&3)==3)&&(t.F[i]&=-32513);if(t.L){if((t.I&fA)!=r0){for(D=B(o.vfb,Tt,6,t.f,16,1),r=0;r<t.f;r++)(t.u[r]&q0)!=0&&(t.u[r]&3)!=3&&(t.u[r]&l1)>>18==1&&(D[r]=true);for(P=0;P<t.f;P++)(t.u[P]&q0)!=0&&(t.u[P]&3)!=3&&(n.on(t,P,1,0),u=true);for(A=0;A<t.f;A++)D[A]&&(n.un(t,A,1,false),n.on(t,A,1,-1),u=true);}t.L=false;}return n.cn(t,1),n.cn(t,2),u;},n.tw=function(){n.Rk(),n.Fq.call(this);},n.uw=function(t,A){n.Rk(),n.Gq.call(this,t,A);},n.vw=function(t){n.Rk(),n.Hq.call(this,t);},x(26,114,fi,n.tw,n.uw,n.vw),s.Yb=function(t){n.aw(this,t);},s.$b=function(t){n.bw(this,t);},s.c=false,o.Q8=N(26),n.zw=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;for(u=new U.Mj(),n.bw(A,1),l=B(o.vfb,Tt,6,A.f,16,1),e=0;e<A.f;e++)if(t.f[e]&&(n.Tl(t.i,e)==7||n.Tl(t.i,e)==8||n.Tl(t.i,e)==16||n.Tl(t.i,e)==34||n.Tl(t.i,e)==52)){for(l[e]=true,Q=0;Q<A.j[e];Q++)if(r=A.i[e][Q],i=A.k[e][Q],y=A.n[e][Q],A.o[r]!=0&&A.o[e]<y){for(l[r]=true,U.sj(u,h.qjb(i)),L=0;L<A.j[r];L++)P=A.i[r][L],l[P]||(l[P]=true,D=A.k[r][L],T=A.n[r][L],A.o[r]+2==y+T&&(U.sj(u,h.qjb(D)),y>=T&&(A.o[P]<T?n.Iw(A,P)&&n.Cw(t,A,P,true,false,l,u):n.Cw(t,A,P,true,true,l,u)),T>=y&&n.Iw(A,e)&&n.Cw(t,A,P,false,false,l,u),t.f[P]&&(n.Tl(t.i,P)==5||n.Tl(t.i,P)==6||n.Tl(t.i,P)==7||n.Tl(t.i,P)==8||n.Tl(t.i,P)==16||n.Tl(t.i,P)==34||n.Tl(t.i,P)==52)&&A.o[P]<T&&(y<=2&&T>=2&&n.Iw(A,e)&&n.Aw(t,A,i,D),y>=2&&T<=2&&n.Iw(A,P)&&n.Aw(t,A,D,i)),U.Ej(u,u.a.length-1)),l[P]=false);U.Ej(u,u.a.length-1),l[r]=false;}l[e]=false;}},n.Aw=function(t,A,e,r){var P;P=new n.Ow(t,A),n.Nw(P,e,n.fm(A,e)==1?2:3),n.Nw(P,r,n.fm(A,r)==2?1:2),t.g[e]=true,t.g[r]=true,U.Ixb(t.e,P)&&U.Kqb(t.d,P);},n.Bw=function(t,A){U.Ixb(t.e,A)&&U.Kqb(t.d,A);},n.Cw=function(t,A,e,r,P,i,D){var u,Q,l,L,y,T,c,b,g,k,v,S;for(c=0;c<A.j[e];c++)if(u=A.i[e][c],!i[u]&&(L=A.k[e][c],v=n.fm(A,L),r&&v>=2||!r&&v<=2)){for(i[u]=true,D.add(h.qjb(L)),b=0;b<A.j[u];b++)if(Q=A.i[u][b],!i[Q]&&(y=A.k[u][b],S=n.fm(A,y),A.o[u]+2==v+S&&(r&&S<=2||!r&&S>=2))){if(i[Q]=true,D.add(h.qjb(y)),t.f[Q]&&(n.Tl(t.i,Q)==5||n.Tl(t.i,Q)==6||n.Tl(t.i,Q)==7||n.Tl(t.i,Q)==8||n.Tl(t.i,Q)==16||n.Tl(t.i,Q)==34||n.Tl(t.i,Q)==52)&&(!r||n.Iw(A,Q))){for(T=new n.Ow(t,A),g=0;g<D.a.length;g++)l=(w.FAb(g,D.a.length),D.a[g]).a,k=g<2?r^(g&1)==0:P^(g&1)==0,k?n.Nw(T,l,n.fm(A,l)==1?2:3):n.Nw(T,l,n.fm(A,l)==2?1:2),t.g[l]=true;U.Ixb(t.e,T)&&U.Kqb(t.d,T);}else n.Cw(t,A,Q,r,P,i,D);U.Ej(D,D.a.length-1),i[Q]=false;}U.Ej(D,D.a.length-1),i[u]=false;}return  false;},n.Dw=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(t.b=B(o.Q7,H,6,t.i.f,15,1),u=B(o.Q7,H,6,t.i.f,15,1),r=B(o.vfb,Tt,6,t.i.g,16,1),y=0,e=0;e<t.i.g;e++)if(!r[e]&&t.g[e])for(++y,t.b[n.Zl(t.i,0,e)]=y,t.b[n.Zl(t.i,1,e)]=y,r[e]=true,l=0;l<2;l++)for(A=n.Zl(t.i,l,e),t.b[A]=y,D=0,Q=0,u[0]=A;D<=Q;){for(L=0;L<n.$o(t.i,u[D]);L++)i=n._o(t.i,u[D],L),!r[i]&&t.g[i]&&(r[i]=true,P=n.Zo(t.i,u[D],L),t.b[P]==0&&(t.b[P]=y,u[++Q]=P));++D;}t.j=y;},n.Ew=function(t){var A,e,r;for(t.n=B(o.Q7,H,6,t.j,15,1),t.k=B(o.Q7,H,6,t.j,15,1),t.o=B(o.Q7,H,6,t.j,15,1),A=0;A<t.i.f;A++)t.b[A]!=0&&(r=t.b[A]-1,t.a!=null&&(t.k[r]+=t.a[A]),t.c!=null&&(t.o[r]+=t.c[A]));for(e=0;e<t.i.g;e++)t.g[e]&&n.fm(t.i,e)==2&&(t.n[t.b[n.Zl(t.i,0,e)]-1]+=2);},n.Fw=function(t){var A,e,r,P;for(r=0;r<t.i.r;r++)for(P=0;P<2;P++)A=n.Zl(t.i,P,r),e=n.Zl(t.i,1-P,r),n.Tl(t.i,A)==1&&n.Ml(t.i,A)>1&&n.Tl(t.i,e)>1&&t.b[e]!=0&&(n.Ml(t.i,A)==2?(t.a==null&&(t.a=B(o.Q7,H,6,t.i.q,15,1)),++t.a[e]):(t.c==null&&(t.c=B(o.Q7,H,6,t.i.q,15,1)),++t.c[e]),n.Um(t.i,A));(t.a!=null||t.c!=null)&&n.vl(t.i);},n.Gw=function(t){var A;for(t.e=new U.Nxb(),t.d=new U.Wqb(),n.Bw(t,new n.Ow(t,t.i)),A=n.fw(t.i);!U.Oqb(t.d);)if(n.Mw(U.Qqb(t.d),A),n.zw(t,A),t.e.a.c>=n.yw){h.Pkb(),""+n.rg(new n.Ng(t.i));break;}},n.Hw=function(t){var A,e,r,P,i,D,u,Q,l,L;if(t.e.a.c==1)return t.i;for(l=n.fw(t.i),n.On(l,true),n.bw(l,7),P=0;P<l.g;P++)t.g[P]&&(l.H[P]=1,l.T=0,n.Ln(l,P,3,true));for(e=0;e<l.f;e++)t.b[e]!=0&&n.tp(t.i,e)<4&&(n.Bo(l,e),l.u[e]&=tr,l.T&=7,n.on(l,e,0,-1));for(u=B(o.Q7,H,6,t.j,15,1),Q=B(o.Q7,H,6,t.j,15,1),r=new n.Ng(l).f,A=0;A<l.f;A++)t.b[A]!=0&&(L=t.b[A]-1,Q[L]<r[A]&&(Q[L]=r[A],u[L]=A));for(i=0;i<t.j;i++)D=""+t.n[i]+"|"+t.k[i]+"|"+t.o[i],n.mn(l,u[i],D);return l;},n.Iw=function(t,A){return t.e[A]-n.tp(t,A)+n.op(t,A)<=0?false:n.Hm(t,A)?true:t.o[A]==0;},n.Jw=function(t){var A,e,r,P,i,D,u,Q;n.bw(t.i,1),u=t.i.f;do--u;while(u>=0&&n.Tl(t.i,u)==1);for(A=0;A<u;A++)if(n.Tl(t.i,A)==1){n.Zn(t.i,A,u);do--u;while(n.Tl(t.i,u)==1);}if(u!=t.i.f-1){for(D=B(o.vfb,Tt,6,t.i.g,16,1),i=0;i<t.i.g;i++)e=n.Zl(t.i,0,i),r=n.Zl(t.i,1,i),(n.Tl(t.i,e)==1||n.Tl(t.i,r)==1)&&(D[i]=true);Q=t.i.g;do--Q;while(Q>=0&&D[Q]);for(P=0;P<Q;P++)if(D[P]){n.$n(t.i,P,Q),D[P]=false;do--Q;while(D[Q]);}}},n.Kw=function(t){var A,e;for(this.i=n.fw(t),n.Jw(this),n.bw(this.i,7),this.g=B(o.vfb,Tt,6,this.i.g,16,1),this.f=B(o.vfb,Tt,6,this.i.f,16,1),A=0;A<this.i.f;A++)e=n.mo(n.Tl(this.i,A))[0],this.f[A]=n.tp(this.i,A)<e;n.Gw(this),n.Dw(this),n.Fw(this),n.Ew(this);},x(241,1,{},n.Kw),s.j=0,n.yw=ur,o.S8=N(241),n.Lw=function(t,A){return Lt.F4(t.a,A.a);},n.Mw=function(t,A){var e,r;for(r=0;r<t.b.i.g;r++)t.b.g[r]&&(e=3&t.a[r>>4]>>2*(r&15),n.Mn(A,r,e==1?1:e==2?t.b.g[r]&&!n.jq(t.b.i,r)?386:2:e==3?4:16));},n.Nw=function(t,A,e){var r,P;r=A>>4,P=2*(A&15),t.a[r]&=~(3<<P),t.a[r]|=e<<P;},n.Ow=function(t,A){var e;for(this.b=t,this.a=B(o.Q7,H,6,(t.i.g+15)/16|0,15,1),e=0;e<t.i.g;e++)this.a[e>>4]|=d.Math.min(3,n.fm(A,e))<<2*(e&15);},x(166,1,{166:1,32:1},n.Ow),s.Sb=function(t){return n.Lw(this,t);},o.R8=N(166),n.Qw=function(t,A,e){var r,P,i,D,u;for(P=0,D=new U.rrb(e);D.a<D.c.a.length;)i=U.qrb(D),w.SAb(i).length!=0&&(u=R.x3(A,i).b,P<u&&(P=u));return r=t.c*Ue*(e.a.length-1)+t.c,new ht.pO(t.i[0].a,t.i[0].b-t.c/2,P,r);},n.Rw=function(t,A,e){return !!t.b&&ht.kO(t.b,A,e);},n.Sw=function(t){var A,e,r,P,i;if(e=h.hkb(t,X1(38)),e==-1)return t;for(P=0,i=new h.Ckb();e!=-1;)h.Bkb(i,(w.MAb(P,e,w.SAb(t).length),w.SAb(t).substr(P,e-P))),w.NAb(e+1,w.SAb(t).length),w.SAb(t).charCodeAt(e+1)==38?(i.a+="&",P=e+2):(A=(w.NAb(e+1,w.SAb(t).length),w.SAb(t).charCodeAt(e+1)),A=A-(A<65?48:A<97?65:97),r=(w.NAb(e+2,w.SAb(t).length),w.SAb(t).charCodeAt(e+2)),r=r-(r<65?48:r<97?65:97),h.zkb(i,16*A+r&e1),P=e+3),e=h.ikb(t,X1(38),P);return h.Bkb(i,(w.NAb(P,w.SAb(t).length+1),w.SAb(t).substr(P))),i.a;},n.Tw=function(t){var A,e;for(e=new h.Ckb(),A=0;A<w.SAb(t).length;A++)switch(w.NAb(A,w.SAb(t).length),w.SAb(t).charCodeAt(A)){case 38:e.a+="&&";break;case 9:e.a+="&09";break;case 10:e.a+="&0A";break;case 32:e.a+="&20";break;default:h.zkb(e,(w.NAb(A,w.SAb(t).length),w.SAb(t).charCodeAt(A)));}return e.a;},n.Uw=function(t,A){var e;return e=n.Vw(t),n.Qw(t,A,e);},n.Vw=function(t){var A,e,r;if(e=new U.Mj(),A=h.hkb(t.e,X1(10)),A==-1)U.sj(e,t.e);else {for(r=0;A!=-1;)U.sj(e,h.rkb(t.e,r,A)),r=A+1,A=h.gkb(t.e,10,r);U.sj(e,h.qkb(t.e,r));}return e;},n.Ww=function(t,A,e){t.i[0].a=A,t.i[0].b=e;},n.Xw=function(t,A){try{t.c=x0(A);}catch(e){if(e=C1(e),!Zt(e,40))throw st(e);}},n.Yw=function(t,A){try{t.d=h.Mib(A,r1,qt);}catch(e){if(e=C1(e),!Zt(e,40))throw st(e);}},n.Zw=function(t,A,e,r){t.e=A,t.c=e,t.d=r;},n.$w=function(t,A){try{t.i[0].a=x0(A);}catch(e){if(e=C1(e),!Zt(e,40))throw st(e);}},n._w=function(t,A){try{t.i[0].b=x0(A);}catch(e){if(e=C1(e),!Zt(e,40))throw st(e);}},n.ax=function(){n.cx.call(this,new ht.aO());},n.bx=function(t){var A,e,r,P;for(n.ax.call(this),A=0;A!=-1&&(e=w.SAb(t).indexOf('="',A),e!=-1);)r=(w.MAb(A+1,e,w.SAb(t).length),w.SAb(t).substr(A+1,e-(A+1))),A=w.SAb(t).indexOf('"',e+2),P=A==-1?(w.NAb(e+1,w.SAb(t).length+1),w.SAb(t).substr(e+1)):(w.MAb(e+1,A,w.SAb(t).length),w.SAb(t).substr(e+1,A-(e+1))),h.akb(r,"text")?this.e=n.Sw(P):h.akb(r,"x")?n.$w(this,P):h.akb(r,"y")?n._w(this,P):h.akb(r,"size")?n.Xw(this,P):h.akb(r,"style")&&n.Yw(this,P);},n.cx=function(t){this.e="",this.c=9,this.d=0,this.i=B(o.Xab,R1,55,1,0,1),this.i[0]=t;},x(94,193,{193:1,94:1},n.ax,n.bx),s.Eb=function(t,A){return this.a=!!this.b&&ht.kO(this.b,t,A),this.a;},s.Fb=function(){this.a=false;},s.Gb=function(){var t;return t=new n.ax(),n.Zw(t,this.e,this.c,this.d),n.Ww(t,this.i[0].a,this.i[0].b),t.g=this.g,t;},s.Hb=function(t,A){return n.Rw(this,t,A);},s.Ib=function(t,A){var e,r,P;for(r=A?A.c*this.c:this.c,R.D3(t,Mt(d.Math.round(r)),(this.d&1)!=0,(this.d&2)!=0),R.F3(t,this.g?fA:R.C3(t)?-1:G0),P=n.Vw(this),this.b=n.Qw(this,t,P),A&&n.xi(A,this.b),e=0;e<P.a.length;e++)R.s3(t,this.b.c,this.b.d+1+r*5/6+r*Ue*e,(w.FAb(e,P.a.length),P.a[e]));},s.Jb=function(t){return n.Uw(this,t);},s.Kb=function(){var t;return t=new h.Kkb(),h.Ikb(t,' text="'+n.Tw(this.e)+'"'),h.Ikb(t,' x="'+this.i[0].a+'"'),h.Ikb(t,' y="'+this.i[0].b+'"'),this.c!=9&&h.Ikb(t,' size="'+M.klb(new M.nlb(this.c,new M.Umb()))+'"'),this.d!=0&&h.Ikb(t,' style="'+this.d+'"'),t.a;},s.Lb=function(){return "text";},s.Mb=function(t){var A,e;A=(e=n.Vw(this),n.Qw(this,t,e)),R.F3(t,R.B3(t)),R.v3(t,A.c,A.d,A.b,A.a);},s.Nb=function(t){n.me(this,t),this.c*=t;},s.Pb=function(t,A){n.qe(this,t,A),this.c=this.f*t;},s.Qb=function(t,A){n.re(this,t,A),this.f=this.c;},s.a=false,s.c=0,s.d=0,s.f=0,o.T8=N(94),n.px=function(t){t.b=new U.Mj(),t.a=new U.Mj();},n.qx=function(t,A){var e,r;return e=n.Xv(t,A),e==-1?-1:(r=t.b.a.length,U.sj(t.b,A),U.rj(t.a,e,h.qjb(r)),r);},n.rx=function(){n.Zv.call(this),n.px(this);},x(282,102,ii,n.rx),s.bc=function(t){var A;return A=n.Yv(this,t),A==-1?-1:U.xj(this.a,A).a;},o.U8=N(282),I.tx=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut;for(Dt=n.Ep(t.e),ot=U.xj(Dt.k,A).length,lt=U.xj(Dt.j,A),ut=U.xj(Dt.k,A),O=true,v=1;v<ot;v++)if(I.Ix(t.b,ut[v])!=I.Ix(t.b,ut[0])){O=false;break;}if(O){for(i=(ot-2)*Vn/ot,S=0;S<ot;S++)I.yx(t,lt[S],ut[S],ut[S==0?ot-1:S-1],i);return;}for(z=B(o.P7,w0,6,ot,15,1),P=0,p=0;p<ot;p++)z[p]=n.Qo(t.e,lt[p])==0?1.9106119386631804:n.Qo(t.e,lt[p])==1?2.094499897810209:Vn,P+=z[p];for(r=((ot-2)*Vn-P)/ot,E=0;E<ot;E++)z[E]+=r;for(L=B(o.P7,w0,6,ot,15,1),q=1;q<ot;q++)L[q]=L[q-1]+Vn-z[q];for(u=B(o.P7,w0,6,ot,15,1),D=0;D<100;D++){for(bt=0,Ut=0,S=0;S<ot;S++)bt+=I.Ix(t.b,ut[S])*d.Math.sin(L[S]),Ut+=I.Ix(t.b,ut[S])*d.Math.cos(L[S]);for(c=n.no(0,0,bt,Ut),g=d.Math.sqrt(bt*bt+Ut*Ut),Qt=-1,tt=0,V=0;V<ot;V++)G=V==0?ot-1:V-1,K=V+1==ot?0:V+1,Q=n.oo(L[V],L[G]),l=n.oo(L[K],L[V]),it=n.oo(z[V],z[K]),u[V]=2*Q-2*l+2*it,b=d.Math.cos(L[V]-wP-c),T=g*b-Di*u[V],d.Math.abs(T)>d.Math.abs(tt)&&(tt=T,Qt=V);y=d.Math.exp(-5*D/100),L[Qt]+=y*tt;}for(k=0;k<ot;k++)G=k==0?ot-1:k-1,e=L[G]+Vn-L[k],e>N1&&(e-=N1),I.yx(t,lt[k],ut[G],ut[k],e);},I.ux=function(t,A){var e,r,P,i,D,u,Q,l,L;for(Q=n.Ep(t.e),l=U.xj(Q.k,A).length,u=U.xj(Q.k,A),L=false,P=0;P<l;P++)if(t.f[u[P]]==0){L=true;break;}if(L){for(e=(180*l-360)/l*.01745329238474369,D=U.xj(Q.j,A),i=0;i<l;i++)I.yx(t,D[i],u[i==0?l-1:i-1],u[i],e);if(l<=4)for(r=0;r<l;r++)t.f[u[r]]+=l;}},I.vx=function(t,A,e,r){var P,i,D,u;for(i=-1,D=-1,u=0;u<n.Lo(t.e,A);u++)if(P=n.Zo(t.e,A,u),P==e||P==r)if(D==-1)D=u;else {i=u;break;}return t.a[A][i][D];},I.wx=function(t,A,e,r){return e<r?t.a[A][r][e]:t.a[A][e][r];},I.xx=function(t,A){var e,r,P,i,D,u,Q;for(r=B(o.vfb,Tt,6,n.$o(t.e,A),16,1),Q=0,P=0;P<n.$o(t.e,A);P++){for(u=0,D=-1,i=0;i<n.$o(t.e,A);i++)r[i]||(e=n._o(t.e,A,i),u<t.f[e]&&(u=t.f[e],D=i));if(u==0)return Q;Q<<=8,Q+=u,r[D]=true;}return Q;},I.yx=function(t,A,e,r,P){var i,D,u,Q;for(i=-1,D=-1,Q=0;Q<n.Lo(t.e,A);Q++)if(u=n._o(t.e,A,Q),u==e||u==r)if(i==-1)i=Q;else {D=Q;break;}t.a[A][D][i]==0&&(t.a[A][D][i]=P,t.d[A]+=P,++t.c[A]);},I.zx=function(t,A,e,r,P){var i,D,u,Q,l,L;for(i=n.Lo(t.e,A),Q=n.$o(t.e,A),l=1;l<i;l++)for(D=l<Q?t.f[n._o(t.e,A,l)]:0,L=0;L<l;L++)t.a[A][l][L]==0&&(u=L<Q?t.f[n._o(t.e,A,L)]:0,(D==e&&u==r||D==r&&u==e)&&(t.a[A][l][L]=P));},I.Ax=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G;for(this.b=A,this.e=t,this.a=B(o.P7,b0,467,this.e.f,0,3),P=0;P<this.e.f;P++)for(this.a[P]=B(o.P7,{467:1,4:1,9:1,5:1,8:1},46,n.Lo(this.e,P),0,2),Q=1;Q<n.Lo(this.e,P);Q++)this.a[P][Q]=B(o.P7,w0,6,Q,15,1);for(this.c=B(o.Q7,H,6,this.e.f,15,1),this.d=B(o.P7,w0,6,this.e.f,15,1),this.f=B(o.Q7,H,6,this.e.g,15,1),V=n.Ep(this.e),T=B(o.vfb,Tt,6,V.j.a.length,16,1),n.at(V,T,B(o.vfb,Tt,6,V.j.a.length,16,1),B(o.Q7,H,6,V.j.a.length,15,1),true),u=3;u<=7;u++)for(v=0;v<V.j.a.length;v++)G=U.xj(V.k,v).length,G==u&&(T[v]?I.tx(this,v):G<=4&&I.ux(this,v));for(y=B(o.vfb,Tt,6,this.e.f,16,1),k=0;k<V.j.a.length;k++)if(T[k])for(p=U.xj(V.j,k),E=0,q=p.length;E<q;++E)S=p[E],y[S]=true;for(i=j(F(o.Q7,1),H,6,15,[0,0,1,3,6,10,15,21]),r=0;r<this.e.f;r++){if(D=n.Lo(this.e,r),D>4){for(Q=1;Q<D;Q++)for(b=0;b<Q;b++)this.a[r][Q][b]=wP;this.c[r]=i[D];continue;}if(this.c[r]!=i[D])if(n.iq(this.e,r)&&(y[r]||n.Uo(this.e,r)<=4&&n.Qo(this.e,r)>0)){if(D>2){if(this.c[r]==1)n.Tl(this.e,r)<=14?e=(ui-this.d[r])/2:e=Qi+(Qi-this.d[r])*.18000000715255737;else if(e=ui-this.d[r],D>3){if(this.c[r]==2){for(c=B(o.vfb,Tt,6,n.Lo(this.e,r),16,1),l=1;l<D;l++)for(g=0;g<l;g++)this.a[r][l][g]!=0&&(c[l]=!c[l],c[g]=!c[g]);for(L=0;L<D;L++)if(c[L]){for(g=L+1;g<D;g++)if(c[g]){this.a[r][g][L]=e;break;}break;}}e=wP;}for(Q=1;Q<D;Q++)for(b=0;b<Q;b++)this.a[r][Q][b]==0&&(this.a[r][Q][b]=e);}}else if(n.iq(this.e,r)&&n.Uo(this.e,r)<=4)switch(I.xx(this,r)){case 771:I.zx(this,r,0,3,2.0653998851776123),I.zx(this,r,0,0,1.9814722631346626);break;case 1028:I.zx(this,r,0,4,1.9797999858856201),I.zx(this,r,0,0,1.94691481878138);break;case 393987:I.zx(this,r,0,3,oi),I.zx(this,r,0,6,oi),I.zx(this,r,3,3,1.7229016938441077);break;case 459779:I.zx(this,r,0,3,gP),I.zx(this,r,0,4,gP),I.zx(this,r,0,7,gP),I.zx(this,r,3,4,1.9322539839360076);break;case 525316:I.zx(this,r,0,4,li),I.zx(this,r,0,8,li),I.zx(this,r,4,4,1.99944913298566);case 394758:I.zx(this,r,0,6,2.526099920272827);break;case 460550:I.zx(this,r,0,6,Li),I.zx(this,r,0,7,Li);break;case 526087:I.zx(this,r,0,7,si),I.zx(this,r,0,8,si);break;case 526344:I.zx(this,r,0,8,2.186300039291382);break;case 50529027:I.zx(this,r,3,3,2.4189000129699707);break;case 67371779:I.zx(this,r,3,4,2.2298998832702637);break;case 67372036:I.zx(this,r,4,4,2.094399929046631);break;case 101057283:I.zx(this,r,3,6,1.839926051241747),I.zx(this,r,3,3,2.9061476191098734);break;case 117834755:I.zx(this,r,3,4,2.812249087174905),I.zx(this,r,3,7,1.7910569124592968),I.zx(this,r,4,6,2.1224948975613245);break;case 134677507:I.zx(this,r,3,4,2.642428360523752),I.zx(this,r,3,8,2.027723514585844),I.zx(this,r,4,7,2.251474717631936);break;case 117900035:I.zx(this,r,3,7,2.109753935530918),I.zx(this,r,3,3,3.1052897491356646);break;case 117900292:I.zx(this,r,4,7,2.090729910747413),I.zx(this,r,4,4,2.551671293386306);break;case 134743044:I.zx(this,r,4,8,2.139250042271712),I.zx(this,r,4,4,2.3520055858942612);}else for(e=n.Tl(this.e,r)>10?Ui:n.Qo(this.e,r)==2?Vn:n.$p(this.e,r,true)?yi:n.Qo(this.e,r)==0?Ui:yi,Q=1;Q<D;Q++)for(b=0;b<Q;b++)this.a[r][Q][b]=e;}},x(382,1,{},I.Ax),o.V8=N(382),I.Hx=function(){I.Hx=f1,I.Fx=j(F(o.vfb,1),Tt,6,16,[false,false,false,false,false,true,true,true,true,false,false,false,false,false,false,true,true]);},I.Ix=function(t,A){return t.a[A];},I.Jx=function(t){I.Hx();var A,e;for(n.bw(t,7),this.a=B(o.P7,w0,6,t.r,15,1),this.b=B(o.P7,w0,6,t.r,15,1),A=0;A<t.r;A++)e=I.Nx(t,A),e==-1?(this.a[A]=I.Ox(t,A),this.b[A]=I.Px(t,A)):(this.a[A]=e==-1?2.000499963760376:I.Dx[e],this.b[A]=e==-1?1:I.Ex[e]);},I.Kx=function(t,A){return A>=t.f?0:A<t.f&&n.ot(t.p,A)&&t.C[A]==6&&t.s[A]!=0?1:t.o[A];},I.Lx=function(t,A,e,r,P,i,D,u,Q){var l,L,y,T,c;return T=r<I.Fx.length&&I.Fx[r]?d.Math.min(3,i):0,c=P<I.Fx.length&&I.Fx[P]?d.Math.min(3,D):0,u=d.Math.min(7,u),Q=d.Math.min(7,Q),l=u+(T<<3)+(d.Math.min(127,r)<<5),L=Q+(c<<3)+(d.Math.min(127,P)<<5),y=e?8:A?8+t:t,(y<<24)+(l<L?(l<<12)+L:(L<<12)+l);},I.Mx=function(t,A,e,r,P,i,D,u,Q,l){var L;return L=I.Rx(I.Lx(t,A,e,r,P,i,D,u,Q)),!l&&L==-1&&(L=I.Rx(I.Lx(t,A,e,r,P,i,D,7,7)),L==-1&&(L=I.Rx(I.Lx(t,A,e,r,P,3,3,7,7)))),L;},I.Nx=function(t,A){var e,r,P,i,D,u;return e=t.D[0][A],r=t.D[1][A],P=t.C[e],i=t.C[r],D=t.j[e],u=t.j[r],I.Mx(n.fm(t,A),A<t.g&&n.pt(t.p,A),A<t.g&&(n.ut(t.p,A)||t.H[A]==8),P,i,I.Kx(t,e),I.Kx(t,r),D,u,false);},I.Ox=function(t,A){var e,r;return e=n.Tl(t,t.D[0][A]),r=n.Tl(t,t.D[1][A]),(e<(I.qz(),I.oz).length?I.oz[e]:e<I.pz.length?ye*I.pz[e]:Qr)+(r<I.oz.length?I.oz[r]:r<I.pz.length?ye*I.pz[r]:Qr);},I.Px=function(t,A){var e,r;return e=n.Tl(t,t.D[0][A]),r=n.Tl(t,t.D[1][A]),(e<(I.qz(),I.oz).length?BP:.125)+(r<I.oz.length?BP:.125);},I.Qx=function(){var t,A,e,r,P,i,D;if(!I.Gx)try{for(t=I.Ey("bondLengthData.txt"),e=Rt.dhb(t),A=e==null?0:h.Mib(e,r1,qt),I.Cx=B(o.Q7,H,6,A,15,1),I.Dx=B(o.P7,w0,6,A,15,1),I.Ex=B(o.P7,w0,6,A,15,1),I.Bx=B(o.Q7,H,6,A,15,1),P=0;P<A;P++)if(D=Rt.dhb(t),D!=null&&(i=h.okb(D,"\\t"),i.length==4))try{I.Cx[P]=h.Mib(i[0],r1,qt),I.Dx[P]=x0(i[1]),I.Ex[P]=x0(i[2]),I.Bx[P]=h.Mib(i[3],r1,qt);}catch(u){if(u=C1(u),Zt(u,40))break;throw st(u);}I.Gx=!0;}catch(u){if(u=C1(u),Zt(u,75))r=u,h.MB(r,(h.Pkb(),h.Okb),"");else throw st(u);}},I.Rx=function(t){var A,e,r,P;for(I.Gx||I.Qx(),P=Z1,r=B1,e=0;e<13;e++){if(A=P>=I.Cx.length||t<I.Cx[P]?-1:t==I.Cx[P]?0:1,A==0)return P;P=A<0?P-r:P+r,r=r/2|0;}return  -1;},x(381,1,{},I.Jx),I.Gx=false,o.W8=N(381),I.Sx=function(t,A){t.q=I.Ry(new I.Uy(t.o,A),t);},I.Tx=function(t,A){var e,r,P,i,D,u,Q,l,L;return e=I.Zx(t,A[0]),r=I.Zx(t,A[1]),P=I.Zx(t,A[2]),i=I.Zx(t,A[3]),Q=new n.oi(r.a-e.a,r.b-e.b,r.c-e.c),l=new n.oi(P.a-r.a,P.b-r.b,P.c-r.c),L=new n.oi(i.a-P.a,i.b-P.b,i.c-P.c),D=new n.oi(Q.b*l.c-Q.c*l.b,-(Q.a*l.c-Q.c*l.a),Q.a*l.b-Q.b*l.a),u=new n.oi(l.b*L.c-l.c*L.b,-(l.a*L.c-l.c*L.a),l.a*L.b-l.b*L.a),-d.Math.atan2(d.Math.sqrt(l.a*l.a+l.b*l.b+l.c*l.c)*(Q.a*u.a+Q.b*u.b+Q.c*u.c),D.a*u.a+D.b*u.b+D.c*u.c);},I.Ux=function(t,A){return I.Xx(t,A),I.Fy(t.q,A.q);},I.Vx=function(t,A){var e;for(e=0;e<A.q;e++)n.ji(t.k[e],A.J[e]);},I.Wx=function(t,A){var e;for(e=0;e<A.k.length;e++)n.ji(t.k[e],A.k[e]);A.j==null?t.j=null:t.j=U.Arb(A.j,A.j.length);},I.Xx=function(t,A){var e;(!t.q||!A.q)&&(e=new I.Ty(t.o),!t.q&&(t.q=I.Ry(e,t)),!A.q&&(A.q=I.Ry(e,A)));},I.Yx=function(t,A){return t.j==null?-1:t.j[A];},I.Zx=function(t,A){return t.k[A];},I.$x=function(t){return t.p==null?t.o.P:t.p;},I._x=function(t,A){return t.k[A].a;},I.ay=function(t,A){return t.k[A].b;},I.by=function(t,A){return t.k[A].c;},I.cy=function(t,A,e){for(t.j==null&&(t.j=B(o.ufb,Yt,6,t.o.r,15,1),U.Grb(t.j));e<0;)e=e+360<<16>>16;for(;e>=360;)e=e-360<<16>>16;t.j[A]=e;},I.dy=function(t,A,e){n.ji(t.k[A],e);},I.ey=function(t,A,e){t.k[A]=e;},I.fy=function(t,A){t.p=A;},I.gy=function(t,A,e){t.k[A].a=e;},I.hy=function(t,A,e){t.k[A].b=e;},I.iy=function(t,A,e){t.k[A].c=e;},I.jy=function(t,A){var e;for(!A&&(A=n.fw(t.o)),A.T&=-249,e=0;e<A.q;e++)n.ji(A.J[e],t.k[e]);return t.p!=null&&n.Un(A,t.p),A;},I.ky=function(t){var A;for(this.o=t,this.k=B(o.h8,En,29,t.q,0,1),A=0;A<t.q;A++)this.k[A]=new n.pi(t.J[A]);this.n=NaN;},I.ly=function(t){I.my.call(this,t,t.o);},I.my=function(t,A){var e;for(this.o=A,this.k=B(o.h8,En,29,t.k.length,0,1),e=0;e<this.k.length;e++)this.k[e]=new n.pi(t.k[e]);t.j!=null&&(this.j=U.Arb(t.j,t.j.length)),this.p=t.p==null||h._jb(t.p)?t.p:h.Yjb(t.p,pn),this.n=NaN;},x(63,1,{63:1,32:1},I.ly),s.Sb=function(t){return I.Ux(this,t);},s.n=0,o.X8=N(63),I.qy=function(t,A){var e,r,P,i,D;return A==null?null:(D=U.eqb(t.c,A),D||(r=w.SAb("<").length,(h.akb(w.SAb(A).substr(w.SAb(A).length-r,r),"<")||(P=w.SAb("-").length,h.akb(w.SAb(A).substr(w.SAb(A).length-P,P),"-")))&&(D=U.eqb(t.c,(i=w.SAb("<").length,h.akb(w.SAb(A).substr(w.SAb(A).length-i,i),"<")?h.rkb(A,0,w.SAb(A).length-1)+">":(e=w.SAb("-").length,h.akb(w.SAb(A).substr(w.SAb(A).length-e,e),"-")?h.rkb(A,0,w.SAb(A).length-1)+"+":A))),D)?(D=new I.mz(D),U.Pwb(t.c,A,D),D):null));},I.ry=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E;t.b|=A;try{for(p=I.Ey("torsionID.txt"),r=(A&1)==0?null:I.Ey("torsionAngle.txt"),T=(A&1)==0?null:I.Ey("torsionRange.txt"),u=(A&1)==0?null:I.Ey("torsionFrequency.txt"),i=(A&2)==0?null:I.Ey("torsionBins.txt"),E=Rt.dhb(p);E!=null;){if(S=U.eqb(t.c,E),S||(S=new I.lz((b=w.SAb("<").length,h.akb(w.SAb(E).substr(w.SAb(E).length-b,b),"<")||(g=w.SAb(">").length,h.akb(w.SAb(E).substr(w.SAb(E).length-g,g),">"))?0:(k=w.SAb("-").length,h.akb(w.SAb(E).substr(w.SAb(E).length-k,k),"-")||(v=w.SAb("+").length,h.akb(w.SAb(E).substr(w.SAb(E).length-v,v),"+"))?1:(c=w.SAb("=").length,h.akb(w.SAb(E).substr(w.SAb(E).length-c,c),"=")?3:2)))),U.Pwb(t.c,E,S)),r)for(e=h.okb(Rt.dhb(r),","),S.a=B(o.ufb,Yt,6,e.length,15,1),l=0;l<e.length;l++)S.a[l]=h.Mib(e[l],Ke,32767)<<16>>16;if(T)for(y=h.okb(Rt.dhb(T),","),S.d=U1(o.ufb,[P0,Yt],[16,6],15,[y.length,2],2),l=0;l<y.length;l++)L=h.gkb(y[l],45,1),S.d[l][0]=h.Mib(h.rkb(y[l],0,L),Ke,32767)<<16>>16,S.d[l][1]=h.Mib(h.qkb(y[l],L+1),Ke,32767)<<16>>16;if(u)for(Q=h.okb(Rt.dhb(u),","),S.c=B(o.ufb,Yt,6,Q.length,15,1),l=0;l<Q.length;l++)S.c[l]=h.Mib(Q[l],-128,127)<<24>>24;if(i)for(P=h.okb(Rt.dhb(i),","),S.b=B(o.M7,Y,6,P.length,15,1),l=0;l<P.length;l++)S.b[l]=h.Mib(P[l],-128,127)<<24>>24;E=Rt.dhb(p);}}catch(q){if(q=C1(q),Zt(q,75))D=q,h.MB(D,(h.Pkb(),h.Okb),"");else throw st(q);}},I.sy=function(t){var A,e,r,P;for(t.a=15,P=(e=new U.Epb(t.c).a.cd().Tb(),new U.Lpb(e));P.a.ed();)r=(A=P.a.fd(),A.jd()),I.jz(r);},I.ty=function(){this.c=new U.Xwb(),this.b=0;},I.uy=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(r=B(o.O7,Ft,6,2,15,1),c=I.xy(t,e),Q=0;Q<2;Q++)if(e[3*Q]!=-1){for(i=2-Q,b=3-3*Q,l=0,T=0;T<3;T++)D=n.Zo(t,e[i],T),D!=c[1-Q]&&t.C[D]!=1&&(e[b]=D,A?r[l++]=I.Tx(A,e):r[l++]=n.el(t,e));return e[b]=-1,I.wy(r);}for(P=B(o.O7,Ft,6,2,15,1),L=0,u=0;u<3;u++)if(g=n.Zo(t,e[1],u),g!=c[0]&&t.C[g]!=1){for(e[0]=g,y=0,T=0;T<3;T++)k=n.Zo(t,e[2],T),k!=c[1]&&t.C[k]!=1&&(e[3]=k,A?P[y++]=I.Tx(A,e):P[y++]=n.el(t,e));r[L++]=I.wy(P);}return e[0]=-1,e[3]=-1,I.wy(r);},I.vy=function(t,A){return A[0]!=-1&&A[3]!=-1?I.Tx(t,A):I.uy(t.o,t,A);},I.wy=function(t){var A,e;return e=(t[1]+t[0])/2,A=d.Math.abs(t[1]-t[0]),A>A1?e:e<0?e+A1:e-A1;},I.xy=function(t,A){var e,r,P;return P=B(o.Q7,H,6,2,15,1),n.Xo(t,A[1],A[2])!=-1?(P[0]=A[2],P[1]=A[1]):(e=B(o.Q7,H,6,16,15,1),r=n.yp(t,e,A[1],A[2],15,null,null),P[0]=e[1],P[1]=e[r-1]),P;},I.yy=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;for(n.bw(t,7),l=0,P=0;P<t.g;P++)n.fm(t,P)==1&&!(P<t.g&&n.pt(t.p,P))&&n.tp(t,t.D[0][P])>1&&n.tp(t,t.D[1][P])>1&&(t.F[P]&64)==0&&!((t.F[P]&128)!=0&&(t.p&&P<t.g?n.ft(t.p,P):0)<=5)&&(A[P]=true,++l);for(D=B(o.Q7,H,6,2,15,1),c=B(o.Q7,H,6,2,15,1),i=null,r=0;r<t.g;r++)if(A[r]&&(i==null||!i[r])){for(e=0,y=0;y<2;y++)for(D[y]=t.D[y][r],c[y]=t.D[1-y][r],g=c[y];n.Qo(t,D[y])==2&&n.tp(t,D[y])==2&&n.Tl(t,D[y])<10;){for(T=0;T<2;T++)if(u=n.Zo(t,D[y],T),u!=c[y]){Q=n._o(t,D[y],T),A[Q]&&n.fm(t,Q)==1&&(A[Q]=false,--l),c[y]=D[y],D[y]=u,++e;break;}if(D[y]==g){e=0;break;}}e!=0&&(A[r]=false,--l,n.tp(t,D[0])>1&&n.tp(t,D[1])>1&&(k=n.Lp(t,c[0],D[0]),v=n.Lp(t,c[1],D[1]),L=k<v?0:1,b=n.Xo(t,c[L],D[L]),i==null&&(i=B(o.vfb,Tt,6,t.g,16,1)),i[b]=true,A[b]=true,++l));}return l;},I.zy=function(t){var A;return A=I.qy(I.py,t),A?I.gz(A):null;},I.Ay=function(t,A,e,r){return I.az(r,t,A),e!=null&&(e[0]=r.f[0],e[1]=r.b[0],e[2]=r.b[1],e[3]=r.f[1]),r.d;},I.By=function(t){var A;return A=I.qy(I.py,t),A?I.hz(A):null;},I.Cy=function(t){var A;return A=I.qy(I.py,t),A?I.fz(A):null;},I.Dy=function(){var t;!I.py&&(I.py=new I.ty()),(I.py.b&1)!=0&&I.py.a!=15&&h.Pkb(),t=1&~I.py.b,t!=0&&(I.ry(I.py,t),t==1&&I.sy(I.py));},I.Ey=function(t){var A,e,r;if(I.oy==null){if(A=_t.cBb("/resources/csd/"+t),A)return I.oy="csd/",new Rt.ehb(new Rt.jhb((r=A,q1.Cnb(),r)));I.oy="cod/";}return new Rt.ehb(new Rt.jhb((e=_t.cBb("/resources/"+I.oy+t),q1.Cnb(),e)));},x(372,1,{},I.ty),s.a=0,s.b=0,o.Y8=N(372),I.Fy=function(t,A){var e,r;for(r=0;r<t.b.length;r++)if(e=d.Math.abs(t.b[r]-A.b[r]),e>or&&e<t.a[r]-or)return e<t.a[r]/2^t.b[r]<A.b[r]?1:-1;return 0;},I.Gy=function(t,A){var e,r;for(r=0;r<t.b.length;r++)if(e=d.Math.abs(t.b[r]-A.b[r]),e>or&&e<t.a[r]-or)return  false;return  true;},I.Hy=function(t,A){this.b=t,this.a=A;},x(256,1,{256:1,32:1},I.Hy),s.Sb=function(t){return I.Fy(this,t);},s.pb=function(){var t,A;for(A=new h.Kkb(),t=0;t<this.b.length;t++)A.a+=t==0?"Torsions: ":", ",h.Ikb(A,(Lt.B4(),Lt.C4(this.b[t],3)+"("+Mt(d.Math.round(this.a[t]+.5))+")"));return A.a;},o.$8=N(256),I.My=function(){I.My=f1,I.Ly=j(F(o.O7,1),Ft,6,15,[N1,A1,In,c0]),I.Ky=j(F(o.Q7,2),E1,7,0,[j(F(o.Q7,1),H,6,15,[0,0,0,2]),j(F(o.Q7,1),H,6,15,[0,0,1,2]),j(F(o.Q7,1),H,6,15,[0,1,1,3]),j(F(o.Q7,1),H,6,15,[2,2,3,2])]);},I.Ny=function(t,A,e,r){var P,i,D,u;if(r==1&&!I.Sy(t,A)){if(n.$o(t.b,A)==3)return  -1;if(n.$o(t.b,A)==3)return P=I.Qy(t,A,e),n.mw(t.b,P[0])==n.mw(t.b,P[1])?P[2]:n.mw(t.b,P[0])==n.mw(t.b,P[2])?P[1]:P[0];}for(D=-1,u=-1,i=0;i<n.$o(t.b,A);i++)P=n.Zo(t.b,A,i),P!=e&&D<n.mw(t.b,P)&&(D=n.mw(t.b,P),u=P);return u;},I.Oy=function(t){var A,e,r,P,i;for(t.a=U1(o.Q7,[E1,H],[7,6],15,[t.d.length,4],2),t.c=U1(o.Q7,[E1,H],[7,6],15,[t.d.length,2],2),t.e=B(o.Q7,H,6,t.d.length,15,1),A=B(o.Q7,H,6,2,15,1),P=0;P<t.d.length;P++){for(i=0;i<2;i++)A[0]=n.Zl(t.b,1-i,t.d[P]),A[1]=n.Zl(t.b,i,t.d[P]),I.Yy(t.b,A[1])&&I.Wy(t.b,A[0],A,null),t.a[P][1+i]=A[1],t.c[P][i]=A[0];e=I.Py(t,t.a[P][1],t.c[P][0]),r=I.Py(t,t.a[P][2],t.c[P][1]),t.e[P]=I.Ky[e][r],t.a[P][0]=I.Ny(t,t.a[P][1],t.c[P][0],e),t.a[P][3]=I.Ny(t,t.a[P][2],t.c[P][1],r);}},I.Py=function(t,A,e){var r;if(n.$o(t.b,A)==2)return 1;if(r=I.Qy(t,A,e),n.$o(t.b,A)==3)return n.mw(t.b,r[0])==n.mw(t.b,r[1])?I.Sy(t,A)?2:1:I.Sy(t,A)?1:0;if(n.$o(t.b,A)==4){if(n.mw(t.b,r[0])==n.mw(t.b,r[1])&&n.mw(t.b,r[0])==n.mw(t.b,r[2]))return 3;if(n.mw(t.b,r[0])==n.mw(t.b,r[1])||n.mw(t.b,r[0])==n.mw(t.b,r[2])||n.mw(t.b,r[1])==n.mw(t.b,r[2]))return 1;}return 0;},I.Qy=function(t,A,e){var r,P,i;for(i=0,r=B(o.Q7,H,6,n.$o(t.b,A)-1,15,1),P=0;P<n.$o(t.b,A);P++)n.Zo(t.b,A,P)!=e&&(r[i++]=n.Zo(t.b,A,P));return r;},I.Ry=function(t,A){var e,r,P;for(P=B(o.P7,w0,6,t.d.length,15,1),r=B(o.P7,w0,6,t.d.length,15,1),e=0;e<t.d.length;e++)P[e]=I.Xy(I.vy(A,t.a[e]),t.e[e]),r[e]=I.Ly[t.e[e]];return new I.Hy(P,r);},I.Sy=function(t,A){return !!(n.Qo(t.b,A)==1&&n.Tl(t.b,A)<10||n.Tp(t.b,A)||n.$p(t.b,A,true));},I.Ty=function(t){I.My(),this.b=t,n.bw(this.b,63),this.d=I.Vy(t),I.Oy(this);},I.Uy=function(t,A){I.My(),this.b=t,n.bw(this.b,63),this.d=A,I.Oy(this);},I.Vy=function(t){I.My();var A,e,r,P,i;for(n.bw(t,7),P=B(o.vfb,Tt,6,t.g,16,1),r=0,e=0;e<t.g;e++)I.Zy(t,e)&&(P[e]=true,++r);for(i=B(o.Q7,H,6,r,15,1),r=0,A=0;A<t.g;A++)P[A]&&(i[r++]=A);return i;},I.Wy=function(t,A,e,r){var P,i,D;for(P=0;P<2;P++)if(i=n.Zo(t,e[1],P),i!=e[0])return i==A||(D=n._o(t,e[1],P),e[0]=e[1],e[1]=i,t.j[i]==1)?false:(r!=null&&!n.Km(t,e[0])&&!n.Km(t,e[1])&&(r[0]=d.Math.max(r[0],D)),t.j[i]==2&&t.o[i]==2&&t.C[i]<=7?I.Wy(t,A,e,r):true);return  false;},I.Xy=function(t,A){var e;for(e=I.Ly[A]/2;t<-e;)t+=N1;for(;t>=e;)t-=I.Ly[A];return t;},I.Yy=function(t,A){return t.j[A]==2&&t.o[A]==2&&t.C[A]<=7;},I.Zy=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;if(n.fm(t,A)!=1||A<t.g&&n.pt(t.p,A)||(t.p&&A<t.g?n.ft(t.p,A):0)==3)return  false;for(e=B(o.Q7,H,6,2,15,1),u=0;u<2;u++)if(e[u]=t.D[u][A],n.Km(t,e[u])||n.tp(t,e[u])<=1)return  false;if(I.Yy(t,e[0])||I.Yy(t,e[1])){for(y=B(o.Q7,H,6,1,15,1),y[0]=A,r=B(o.Q7,E1,7,2,0,2),Q=0;Q<2;Q++)if(I.Yy(t,e[Q])&&(r[Q]=B(o.Q7,H,6,2,15,1),r[Q][0]=e[1-Q],r[Q][1]=e[Q],!I.Wy(t,e[1-Q],r[Q],y)))return  false;if(A!=y[0])return  false;for(l=0;l<2;l++)r[l]!=null&&(e[l]=r[l][1]);}for(D=0;D<2;D++){if(i=n.$o(t,e[D]),i==1)return  false;for(T=0,L=0;L<i;L++)P=n.Zo(t,e[D],L),(t.u[P]&r0)!=0||++T;if(T<2)return  false;}return  true;},x(254,1,{},I.Ty,I.Uy),o.Z8=N(254),I._y=function(){I._y=f1,I.$y=j(F(o.Q7,2),E1,7,0,[j(F(o.Q7,1),H,6,15,[0,0,1]),j(F(o.Q7,1),H,6,15,[0,2,3]),j(F(o.Q7,1),H,6,15,[1,3,3])]);},I.az=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt;if(n._v(t.c),t.d=null,n.bw(A,63),n.fm(A,e)!=1||e<A.g&&n.pt(A.p,e)||n.Tl(A,A.D[0][e])==1||n.Tl(A,A.D[1][e])==1||(lt=(A.F[e]&128)!=0,lt&&(A.p&&e<A.g?n.ft(A.p,e):0)<6))return  false;for(P=B(o.vfb,Tt,6,A.f,16,1),t.a=0,G=0;G<2;G++){for(t.b[G]=A.D[G][e],t.e[G]=A.D[1-G][e];n.Qo(A,t.b[G])==2&&n.tp(A,t.b[G])==2&&n.Tl(A,t.b[G])<10;)for(ut=0;ut<n.$o(A,t.b[G]);ut++)if(D=n.Zo(A,t.b[G],ut),D!=t.e[G]&&A.C[D]!=1){if(A.j[D]==1||t.a==16)return  false;P[t.b[G]]=true,t.e[G]=t.b[G],t.b[G]=D,++t.a;break;}if(Dt=n.tp(A,t.b[G]),Dt>4||Dt==1)return  false;P[t.b[G]]=true;}for(K=0;K<2;K++)for(ut=0;ut<n.$o(A,t.b[K]);ut++)D=n.Zo(A,t.b[K],ut),A.C[D]!=1&&(P[D]=true);for(t.g=B(o.Q7,H,6,A.f,15,1),n.Co(A,t.c,P,true,t.g),O=0;O<t.g.length;O++)t.g[O]!=-1&&(t.i[t.g[O]]=O);if(n.On(t.c,true),lt&&(i=n.Xo(t.c,t.g[t.b[0]],t.g[t.b[1]]),i!=-1)){for(n.Ln(t.c,i,256,true),Et=(n.bw(A,7),A.p),It=0;It<Et.j.a.length;It++)if(n.rt(Et,It,e)){for(tt=0;tt<2;tt++)for(ut=0;ut<n.$o(A,t.b[tt]);ut++)if(D=n.Zo(A,t.b[tt],ut),D!=t.e[tt]&&n.qt(Et,It,D)&&A.C[D]!=1){n.Ln(t.c,n.Xo(t.c,t.g[t.b[tt]],t.g[D]),256,true);break;}}}for(Qt=0;Qt<2;Qt++){for(n.$p(A,t.b[Qt],true)&&n.vn(t.c,t.g[t.b[Qt]],NA,true),Q=false,ut=0;ut<n.$o(A,t.b[Qt]);ut++)D=n.Zo(A,t.b[Qt],ut),D!=t.e[Qt]&&A.C[D]!=1&&(v=n.Xo(t.c,t.g[t.b[Qt]],t.g[D]),n.im(t.c,v)==8?Q=true:A.C[D]==6&&!n.Tp(A,t.b[Qt])&&(k=D<A.f&&n.ot(A.p,D)?2:4,n.vn(t.c,t.g[D],k,true)),u=n._o(A,t.b[Qt],ut),Ht=A.p&&u<A.g?n.ft(A.p,u):0,(Ht==3||Ht==4)&&n.Ln(t.c,v,Ht<<17,true),(u<A.g&&n.pt(A.p,u)||n.ap(A,t.b[Qt],ut)==2)&&(Dt=n.tp(A,D),q=Dt==3,!q&&Dt==2&&(A.u[D]&V0)==0&&(q=n.Mp(A,t.b[1-Qt],u)!=-1),q?n.vn(t.c,t.g[D],Rn,true):u<A.g&&n.pt(A.p,u)&&n.vn(t.c,t.g[D],Mn,true)),n.ap(A,t.b[Qt],ut)==1&&(n.tp(A,D)==4?n.vn(t.c,t.g[D],ln,true):A.C[D]==6&&n.vn(t.c,t.g[D],j0,true)));Q||(n.Tp(A,t.b[Qt])?n.vn(t.c,t.g[t.b[Qt]],2,true):n.vn(t.c,t.g[t.b[Qt]],4,true));}for(n.bw(t.c,191),V=0;V<2;V++)if(S=t.g[t.b[V]],n.Nl(t.c,S)==3){if(n.Nl(A,t.b[V])==3)return  false;ot=n.Ro(t.c,S),n.Mn(t.c,ot,257),n.Zl(t.c,0,ot)!=S&&(n.Dn(t.c,1,ot,n.Zl(t.c,0,ot)),n.Dn(t.c,0,ot,S)),n.bw(t.c,191);}if(T=t.g[t.b[0]],c=t.g[t.b[1]],wt=t.g[t.e[0]],ct=t.g[t.e[1]],l=n.Hl(t.c,T),L=n.Hl(t.c,c),n.zm(t.c,T)&&n.zm(t.c,c)&&(l!=0||L!=0)&&(l!=L||n.Gl(t.c,T)!=n.Gl(t.c,c)))return  false;if(y=false,n.zm(t.c,T)&&l!=0&&(n.on(t.c,T,0,-1),y=true),n.zm(t.c,c)&&L!=0&&(n.on(t.c,c,0,-1),y=true),y&&n.bw(t.c,191),bt=n.mw(t.c,T),Ut=n.mw(t.c,c),bt<Ut&&(Nt=t.b[0],t.b[0]=t.b[1],t.b[1]=Nt,Nt=t.e[0],t.e[0]=t.e[1],t.e[1]=Nt,Nt=T,T=c,c=Nt,Nt=wt,wt=ct,ct=Nt),it=false,(n.zm(t.c,T)||n.zm(t.c,c))&&(n.zm(t.c,T)?it=n.cw(t.c,T)==1:n.zm(t.c,c)&&(it=n.cw(t.c,c)==1),it)){for(r=0;r<t.c.q;r++)n.zn(t.c,r,-n.Ql(t.c,r));n.bw(t.c,191);}return b=I.cz(t,T,wt),g=I.cz(t,c,ct),t.f[0]=b==-1?-1:t.i[b],t.f[1]=g==-1?-1:t.i[g],z=n.jw(t.c),z==null?false:(p=I.bz(t,T,wt),E=I.bz(t,c,ct),p==0&&E==0&&(t.c.I&fA)==r0?xt=2:xt=I.$y[p][E],Kt=xt==0?it?"<":">":xt==1?it?"-":"+":xt==3?"=":"",t.d=z+Kt,true);},I.bz=function(t,A,e){var r,P,i,D;if(n.$o(t.c,A)==2)return 1;if(r=I.dz(t,A,e),n.$o(t.c,A)==3)return n.mw(t.c,r[0])==n.mw(t.c,r[1])?n.Qo(t.c,A)==1&&n.Tl(t.c,A)<10||n.Tp(t.c,A)||n.$p(t.c,A,true)?2:1:n.Qo(t.c,A)==1&&n.Tl(t.c,A)<10||n.Tp(t.c,A)||n.$p(t.c,A,true)?1:0;if(n.$o(t.c,A)==4){for(P=0;P<r.length;P++)for(D=n.mw(t.c,r[P]),i=P+1;i<r.length;i++)if(D==n.mw(t.c,r[i]))return 1;}return 0;},I.cz=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T;for(L=-1,y=-1,T=-1,i=B(o.vfb,Tt,6,n.$o(t.c,A),16,1),Q=0;Q<n.$o(t.c,A);Q++)if(!i[Q]&&(P=n.Zo(t.c,A,Q),P!=e&&(D=n.mw(t.c,P),y<D))){for(u=false,l=Q+1;l<n.$o(t.c,A);l++)if(r=n.Zo(t.c,A,l),r!=e&&n.mw(t.c,r)==D){if(i[l]=true,u)return P;u=true;}u?T=P:(y=D,L=P);}return L==-1&&(n.Qo(t.c,A)==1&&n.Tl(t.c,A)<10||n.Tp(t.c,A)||n.$p(t.c,A,true))?T:L;},I.dz=function(t,A,e){var r,P,i;for(i=0,r=B(o.Q7,H,6,n.$o(t.c,A)-1,15,1),P=0;P<n.$o(t.c,A);P++)n.Zo(t.c,A,P)!=e&&(r[i++]=n.Zo(t.c,A,P));return r;},I.ez=function(){I._y(),this.c=new n.uw(24,29),this.i=B(o.Q7,H,6,24,15,1),this.b=B(o.Q7,H,6,2,15,1),this.e=B(o.Q7,H,6,2,15,1),this.f=B(o.Q7,H,6,2,15,1);},x(250,1,{},I.ez),s.a=0,o._8=N(250),I.fz=function(t){var A,e,r,P,i,D,u,Q,l;switch(A=null,t.e){case 1:for(A=B(o.ufb,Yt,6,2*t.a.length,15,1),r=0;r<t.a.length;r++)A[r]=t.a[r],A[t.a.length+r]=180+t.a[r]<<16>>16;return A;case 2:for(P=t.a[0]==0?1:0,i=t.a[t.a.length-1]==180?t.a.length-1:t.a.length,l=i-P,A=B(o.ufb,Yt,6,t.a.length+l,15,1),D=0;D<t.a.length;D++)A[D]=t.a[D];for(u=P;u<i;u++)A[A.length-1-u+P]=360-t.a[u]<<16>>16;return A;case 3:for(P=t.a[0]==0?1:0,i=t.a[t.a.length-1]==90?t.a.length-1:t.a.length,l=i-P,A=B(o.ufb,Yt,6,2*t.a.length+2*l,15,1),Q=0;Q<t.a.length;Q++)A[Q]=t.a[Q],A[t.a.length+l+Q]=180+t.a[Q]<<16>>16;for(e=P;e<i;e++)A[t.a.length+l-1-e+P]=180-t.a[e]<<16>>16,A[A.length-1-e+P]=360-t.a[e]<<16>>16;return A;default:return t.a;}},I.gz=function(t){var A,e,r,P,i,D,u,Q,l,L;switch(A=null,L=t.c.length,t.e){case 1:for(A=B(o.ufb,Yt,6,2*L,15,1),r=0;r<L;r++)A[r]=t.c[r],A[L+r]=t.c[r];return A;case 2:for(P=t.a[0]==0?1:0,i=t.a[L-1]==180?L-1:L,l=i-P,A=B(o.ufb,Yt,6,L+l,15,1),D=0;D<L;D++)A[D]=t.c[D];for(u=P;u<i;u++)A[A.length-1-u+P]=t.c[u];return A;case 3:for(P=t.a[0]==0?1:0,i=t.a[L-1]==90?L-1:L,l=i-P,A=B(o.ufb,Yt,6,2*L+2*l,15,1),Q=0;Q<L;Q++)A[Q]=t.c[Q],A[L+l+Q]=t.c[Q];for(e=P;e<i;e++)A[L+l-1-e+P]=t.c[e],A[A.length-1-e+P]=t.c[e];return A;default:return t.c;}},I.hz=function(t){var A,e,r,P,i,D,u,Q,l,L;switch(A=null,L=t.d.length,t.e){case 1:for(A=U1(o.ufb,[P0,Yt],[16,6],15,[2*L,2],2),r=0;r<L;r++)A[r][0]=t.d[r][0],A[r][1]=t.d[r][1],A[L+r][0]=180+t.d[r][0]<<16>>16,A[L+r][1]=180+t.d[r][1]<<16>>16;return A;case 2:for(P=t.a[0]==0?1:0,i=t.a[L-1]==180?L-1:L,l=i-P,A=U1(o.ufb,[P0,Yt],[16,6],15,[L+l,2],2),D=0;D<L;D++)A[D][0]=t.d[D][0],A[D][1]=t.d[D][1];for(u=P;u<i;u++)A[A.length-1-u+P][0]=360-t.d[u][1]<<16>>16,A[A.length-1-u+P][1]=360-t.d[u][0]<<16>>16;return A;case 3:for(P=t.a[0]==0?1:0,i=t.a[L-1]==90?L-1:L,l=i-P,A=U1(o.ufb,[P0,Yt],[16,6],15,[2*L+2*l,2],2),Q=0;Q<L;Q++)A[Q][0]=t.d[Q][0],A[Q][1]=t.d[Q][1],A[L+l+Q][0]=180+t.d[Q][0]<<16>>16,A[L+l+Q][1]=180+t.d[Q][1]<<16>>16;for(e=P;e<i;e++)A[L+l-1-e+P][0]=180-t.d[e][1]<<16>>16,A[L+l-1-e+P][1]=180-t.d[e][0]<<16>>16,A[A.length-1-e+P][0]=360-t.d[e][1]<<16>>16,A[A.length-1-e+P][1]=360-t.d[e][0]<<16>>16;return A;default:return t.d;}},I.iz=function(t,A,e,r,P,i){var D,u,Q,l,L,y;for(t.a[e]=A<<16>>16,t.c[e]=t.c[e]+t.c[r]<<16>>16,t.d[e][0]=P<<16>>16,t.d[e][1]=i<<16>>16,y=t.a.length-1,Q=B(o.ufb,Yt,6,y,15,1),l=B(o.ufb,Yt,6,y,15,1),L=B(o.ufb,P0,16,y,0,2),u=0,D=0;D<t.a.length;D++)D!=r&&(Q[u]=t.a[D],l[u]=t.c[D],L[u]=t.d[D],++u);t.a=Q,t.c=l,t.d=L;},I.jz=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T;for(A=0;t.a.length!=0;){if(t.a.length==1){if(t.e==1||t.e==0)break;if(t.e==2&&(t.a[0]==0||t.a[0]==180))break;if(t.e==3&&(t.a[0]==0||t.a[0]==90))break;}for(T=t.a[0]!=0&&(t.e==2||t.e==3)?-1:0,y=-1,L=qt,D=T;D<t.a.length;D++)Q=D+1,e=D>=0?t.a[D]:-t.a[0],r=Q<t.a.length?t.a[Q]:t.e==1?t.a[0]+180:t.e==2?360-t.a[t.a[D]==180?D-1:D]:t.e==3?180-t.a[t.a[D]==90?D-1:D]:t.a[0]+360,L>r-e&&(L=r-e,y=D);if(L>15)break;i=y,u=i+1,i==-1?(A+=t.c[0],t.a[0]=0,t.c[0]=t.c[0]*2<<16>>16,t.d[0][0]=-t.d[0][1]):u<t.a.length?(l=I.kz(t,i,u,t.a[u]),I.iz(t,l,i,u,t.d[i][0],t.d[u][1])):t.e==1?(u=0,l=I.kz(t,i,u,t.a[u]+180),l<=180?I.iz(t,l,i,u,t.d[i][0],t.d[u][1]+180):I.iz(t,l-180,u,i,t.d[u][0],t.d[i][1]-180)):t.e==2?(u=t.a[i]==180?i-1:i,t.a[i]<180?(A+=t.c[i],t.a[i]=180,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=360-t.d[i][0]<<16>>16):(h.Pkb(),l=I.kz(t,i,u,t.a[u]),I.iz(t,l,u,i,t.d[u][0],t.d[i][1]))):t.e==3?(u=t.a[i]==90?i-1:i,t.a[i]<90?(A+=t.c[i],t.a[i]=90,t.c[i]=t.c[i]*2<<16>>16,t.d[i][1]=180-t.d[i][0]<<16>>16):(h.Pkb(),l=I.kz(t,i,u,t.a[u]),I.iz(t,l,u,i,t.d[u][0],t.d[i][1]))):(u=0,l=I.kz(t,i,u,t.a[u]+360),l<=360?I.iz(t,l,i,u,t.d[i][0],t.d[u][1]+360):I.iz(t,l-360,u,i,t.d[u][0],t.d[i][1]-360));}if(A!=0)for(P=0;P<t.c.length;P++)t.c[P]=Mt(d.Math.round(100*t.c[P]/(100+A)))<<16>>16;},I.kz=function(t,A,e,r){return Mt(d.Math.round((t.a[A]*t.c[A]+r*t.c[e])/(t.c[A]+t.c[e])));},I.lz=function(t){this.e=t;},I.mz=function(t){var A;if(this.e=t.e,t.a!=null)for(this.a=B(o.ufb,Yt,6,t.a.length,15,1),A=0;A<this.a.length;A++)this.a[A]=360-t.a[this.a.length-A-1]<<16>>16;if(t.d!=null)for(this.d=U1(o.ufb,[P0,Yt],[16,6],15,[t.d.length,2],2),A=0;A<this.d.length;A++)this.d[A][0]=360-t.d[this.a.length-A-1][1]<<16>>16,this.d[A][1]=360-t.d[this.a.length-A-1][0]<<16>>16;if(t.c!=null)for(this.c=B(o.ufb,Yt,6,t.c.length,15,1),A=0;A<this.c.length;A++)this.c[A]=t.c[this.c.length-A-1];if(t.b!=null)for(this.b=B(o.M7,Y,6,t.b.length,15,1),A=0;A<this.b.length;A++)this.b[A]=t.b[this.b.length-A-1];},x(194,1,{194:1},I.lz,I.mz),s.e=0,o.a9=N(194),I.nz=function(t,A){var e,r,P,i;e=A[1],r=A[2],P=t.j[e]-1,i=t.j[r]-1,P>3||i>3?(this.b=B(o.ufb,Yt,6,4,15,1),this.b[0]=45,this.b[1]=135,this.b[2]=225,this.b[2]=315,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[4,2],2),this.c[0][0]=30,this.c[0][1]=60,this.c[1][0]=120,this.c[1][1]=150,this.c[2][0]=210,this.c[2][1]=240,this.c[2][0]=300,this.c[2][1]=330,this.a=B(o.ufb,Yt,6,4,15,1),this.a[0]=25,this.a[1]=25,this.a[2]=25,this.a[3]=25):(t.o[e]==0||t.C[e]>9)&&(t.o[r]==0||t.C[r]>9)?P==3&&i==3||P==3&&i==2||P==3&&i==1||P==2&&i==3||P==1&&i==3||P==2&&i==2&&(A[0]!=-1||A[3]!=-1)?(this.b=B(o.ufb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=B(o.ufb,Yt,6,3,15,1),this.a[0]=33,this.a[1]=33,this.a[2]=33):P==1&&i==2&&A[3]==-1||P==2&&i==1&&A[0]==-1?(this.b=B(o.ufb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=B(o.ufb,Yt,6,3,15,1),this.a[0]=40,this.a[1]=20,this.a[2]=40):(P==1&&i==1||P==1&&i==2&&A[3]!=-1||P==2&&i==1&&A[0]!=-1||P==2&&i==2&&A[0]==-1&&A[3]==-1)&&(this.b=B(o.ufb,Yt,6,3,15,1),this.b[0]=60,this.b[1]=180,this.b[2]=300,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=45,this.c[0][1]=75,this.c[1][0]=165,this.c[1][1]=195,this.c[2][0]=285,this.c[2][1]=315,this.a=B(o.ufb,Yt,6,3,15,1),this.a[0]=25,this.a[1]=50,this.a[2]=25):(t.o[e]==0||t.C[e]>9)&&t.o[r]==1||(t.o[r]==0||t.C[r]>9)&&t.o[e]==1?P==3||i==3?(this.b=B(o.ufb,Yt,6,6,15,1),this.b[0]=0,this.b[1]=60,this.b[2]=120,this.b[3]=180,this.b[4]=240,this.b[5]=300,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=45,this.c[1][1]=75,this.c[2][0]=105,this.c[2][1]=135,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=225,this.c[4][1]=255,this.c[5][0]=285,this.c[5][1]=315,this.a=B(o.ufb,Yt,6,6,15,1),this.a[0]=16,this.a[1]=16,this.a[2]=16,this.a[3]=16,this.a[4]=16,this.a[5]=16):P==1&&i==1?(this.b=B(o.ufb,Yt,6,2,15,1),this.b[0]=120,this.b[1]=240,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=105,this.c[0][1]=135,this.c[1][0]=225,this.c[1][1]=255,this.a=B(o.ufb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[e]==1&&P==2&&i==1||t.o[r]==1&&i==2&&P==1?(this.b=B(o.ufb,Yt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=B(o.ufb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):t.o[e]==1&&P==1&&i==2&&A[3]==-1||t.o[r]==1&&i==1&&P==2&&A[0]==-1?(this.b=B(o.ufb,Yt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=B(o.ufb,Yt,6,3,15,1),this.a[0]=60,this.a[1]=20,this.a[2]=20):t.o[e]==1&&P==1&&i==2&&A[3]!=-1||t.o[r]==1&&i==1&&P==2&&A[0]!=-1?(this.b=B(o.ufb,Yt,6,3,15,1),this.b[0]=0,this.b[1]=120,this.b[2]=240,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[3,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=105,this.c[1][1]=135,this.c[2][0]=225,this.c[2][1]=255,this.a=B(o.ufb,Yt,6,3,15,1),this.a[0]=20,this.a[1]=40,this.a[2]=40):P==2&&i==2&&(A[0]==-1||A[3]==-1?(this.b=B(o.ufb,Yt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=B(o.ufb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50):(this.b=B(o.ufb,Yt,6,2,15,1),this.b[0]=90,this.b[1]=270,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=75,this.c[0][1]=105,this.c[1][0]=255,this.c[1][1]=285,this.a=B(o.ufb,Yt,6,2,15,1),this.a[0]=50,this.a[1]=50)):t.o[e]==1&&t.o[r]==1?P==1&&i==1?(this.b=B(o.ufb,Yt,6,2,15,1),this.b[0]=0,this.b[1]=180,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[2,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=165,this.c[1][1]=195,this.a=B(o.ufb,Yt,6,2,15,1),this.a[0]=10,this.a[1]=90):(this.b=B(o.ufb,Yt,6,6,15,1),this.b[0]=0,this.b[1]=50,this.b[2]=130,this.b[3]=180,this.b[4]=230,this.b[5]=310,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[6,2],2),this.c[0][0]=-15,this.c[0][1]=15,this.c[1][0]=35,this.c[1][1]=65,this.c[2][0]=115,this.c[2][1]=145,this.c[3][0]=165,this.c[3][1]=195,this.c[4][0]=215,this.c[4][1]=245,this.c[5][0]=295,this.c[5][1]=325,this.a=B(o.ufb,Yt,6,6,15,1),this.a[0]=40,this.a[1]=5,this.a[2]=5,this.a[3]=40,this.a[4]=5,this.a[5]=5):(this.b=B(o.ufb,Yt,6,1,15,1),this.b[0]=180,this.c=U1(o.ufb,[P0,Yt],[16,6],15,[1,2],2),this.c[0][0]=165,this.c[0][1]=195,this.a=B(o.ufb,Yt,6,1,15,1),this.a[0]=100);},x(251,1,{},I.nz),o.b9=N(251),I.qz=function(){I.qz=f1,I.pz=j(F(o.P7,1),w0,6,15,[1,Te,kP,2.119999885559082,1.9800000190734863,1.909999966621399,1.7699999809265137,lr,1.5,dP,1.5800000429153442,2.5,2.509999990463257,2.25,2.190000057220459,mP,1.8899999856948853,Ti,1.8300000429153442,2.7300000190734863,2.619999885559082,2.5799999237060547,hi,ci,Lr,Lr,vP,bi,bi,2.380000114440918,2.390000104904175,FP,jP,1.8799999952316284,Ti,wi,2.25,3.2100000381469727,2.8399999141693115,2.75,2.5199999809265137,2.559999942779541,Lr,vP,hi,vP,2.1500000953674316,gi,Bi,2.430000066757202,ci,ki,1.9900000095367432,2.0399999618530273,2.059999942779541,3.4800000190734863,3.0299999713897705,2.9800000190734863,di,mi,2.950000047683716,mi,2.9000000953674316,vi,pP,Fi,vi,ji,pP,Fi,pi,2.740000009536743,2.630000114440918,gi,2.569999933242798,Bi,2.4800000190734863,2.4100000858306885,jP,FP,Lr,ki,2.5999999046325684,2.5399999618530273,2.5,2.5,2.5,2.5,2.5,pi,2.930000066757202,di,2.7100000381469727,2.819999933242798,ji,pP,Mi,3.4000000953674316,Mi,2.700000047683716]),I.oz=j(F(o.P7,1),w0,6,15,[.25,MP,.46000000834465027,Ri,1.0199999809265137,he,.75,.7099999785423279,.6299999952316284,.6399999856948853,.9599999785423279,Ci,Si,1.2599999904632568,RP,Hi,1.0299999713897705,.9900000095367432,Ei,qi,Vi,1.4800000190734863,CP,Ii,Oi,1.190000057220459,RP,Hi,1.100000023841858,Te,Te,xi,sr,sr,RP,1.1399999856948853,sr,Ni,SP,HP,1.5399999618530273,Gi,1.3799999952316284,zi,1.25,1.25,Te,Si,Ji,dP,Ue,Ue,CP,Ri,1.350000023841858,FP,qi,Qr,HP,Ki,1.7400000095367432,Zi,Yi,Ur,Xi,Ur,_i,lr,EP,1.6399999856948853,ce,Wi,1.5199999809265137,dP,$i,1.309999942779541,ai,Oi,1.2300000190734863,xi,1.4199999570846558,1.5,Ji,1.5099999904632568,t9,Gi,t9,2.2300000190734863,2.009999990463257,wi,1.75,Xi,ce,Vi,Yi,lr,lr,Ur,Ur,EP,_i,Zi,Ki,1.6100000143051147,A9,1.4900000095367432,kP,1.409999966621399,Ii,ai,zi,sr,$i,CP,kP,Wi,1.75,EP,A9]);},Xt.sz=function(){Xt.sz=f1,Xt.rz=n.po("X",32);},Xt.tz=function(t,A){Xt.sz();var e,r,P,i,D;for(P=0;P<t.q;P++)for(D=n.fw(t),n.Cn(D,P,Xt.rz),n.bw(D,15),r=0;r<D.f;r++)(D.u[r]&q0)!=0&&n.Ip(D,r)==-1&&(i=(n.bw(D,7),D.o[r]==2&&D.j[r]==2?n.nq(D,r,false):n.pq(D,r,false)),i!=-1&&(t.H[i]=257,t.T=0,t.D[1][i]==r&&(e=t.D[0][i],t.D[0][i]=r,t.T=0,t.D[1][i]=e,t.T=0),n.on(t,r,A,0)));},Xt.uz=function(t){Xt.sz();var A,e,r,P;for(Xt.tz(t,1),r=t.q,e=B(o.Ocb,yt,2,r,6,1),A=0;A<r;A++)P=n.fw(t),n.mn(P,A,(n.Rk(),n.Lk)[P.C[A]]+"*"),P.C[A]==1?n.Cn(P,A,Xt.rz):n.tn(P,A,P.A[A]+5),Xt.vz(P),e[A]=n.rg(new n.Og(P,8));return e;},Xt.vz=function(t){var A;for(n.bw(t,15),A=0;A<t.q;A++)(t.u[A]&3)!=0&&n.on(t,A,1,0);},Xt.rz=0,Xt.wz=function(t,A,e){var r,P;for(r=U1(o.Ocb,[b0,yt],[13,2],6,[t.f,A],2),n.bw(t,7),P=0;P<t.f;P++)r[P]=Xt.xz(t,P,A,e);return r;},Xt.xz=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g;for(Q=new n.uw(t.f,t.g),L=new U.Zxb(),c=0,T=0,D=B(o.vfb,Tt,6,t.f,16,1),i=B(o.Q7,H,6,t.f,15,1),g=0;g<e&&T<t.f;g++){if(T==0)i[0]=A,D[A]=true,T=1;else {for(b=T,l=c;l<T;l++)for(P=i[l],y=0;y<t.j[P];y++)if(u=t.i[P][y],!D[u])switch(r){case 0:D[u]=true,i[b++]=u;break;case 1:Xt.zz(t,P)&&Xt.zz(t,u)||(D[u]=true,i[b++]=u);}c=T,T=b;}n.Co(t,Q,D,true,null),U.Wxb(L,n.rg(new n.Og(Q,8)));}return U.Yxb(L,B(o.Ocb,yt,2,L.a.a.length,6,1));},Xt.yz=function(t,A,e){var r,P,i,D,u,Q;for(u=n.Pq(new n._q(true),t),P=-1,i=0;i<u.q;i++)if(r=u.t==null||u.t[i]==null?null:h.$jb((D=u.t[i],q1.Cnb(),D)),r!=null&&(Q=w.SAb("*").length,h.akb(w.SAb(r).substr(w.SAb(r).length-Q,Q),"*"))){P=i;break;}return P>=0?Xt.xz(u,P,A,e):B(o.Ocb,yt,2,0,6,1);},Xt.zz=function(t,A){return !(t.C[A]!=6||t.s[A]!=0||n.op(t,A)+t.j[A]!=4);},Xt.Kz=function(){Xt.Kz=f1,Xt.Az=d.Math.cos(Sn),Xt.Fz=d.Math.sin(Sn),Xt.Cz=d.Math.cos(qP),Xt.Hz=d.Math.sin(qP),Xt.Ez=d.Math.cos(c0),Xt.Jz=d.Math.sin(c0),Xt.Bz=d.Math.cos(be),Xt.Gz=d.Math.sin(be),Xt.Dz=d.Math.cos(lP),Xt.Iz=d.Math.sin(lP),d.Math.cos(n9),d.Math.sin(n9);},Xt.Lz=function(t){Xt.Kz();var A,e;for(n.bw(t,1),e=t.f,A=0;A<e;A++)Xt.Mz(t,A);},Xt.Mz=function(t,A){Xt.Kz();var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut;switch(G=n.op(t,A),lt=t.J[A].a,ut=t.J[A].b,G){case 1:{if(O=t.j[A],O==0?(L=n.Vo(t,true),c=0):(L=lt-n.Ql(t,t.i[A][0]),c=ut-n.Rl(t,t.i[A][0])),O==1)p=n.Tk(t,lt+Xt.Dz*L+Xt.Iz*c,ut-Xt.Iz*L+Xt.Dz*c,0);else if(O==2)L=lt-.5*(n.Ql(t,t.i[A][0])+n.Ql(t,t.i[A][1])),c=ut-.5*(n.Rl(t,t.i[A][0])+n.Rl(t,t.i[A][1])),p=n.Tk(t,lt+L,ut+c,0);else if(O==3){for(Qt=t.i[A][0],v=1;v<3;v++)u=t.k[A][v],(t.H[u]==129||t.H[u]==257)&&(Qt=t.i[A][v]);if(e=d.Math.abs(n.oo(n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][0]),n.Rl(t,t.i[A][0])),n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][1]),n.Rl(t,t.i[A][1])))),r=d.Math.abs(n.oo(n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][0]),n.Rl(t,t.i[A][0])),n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][2]),n.Rl(t,t.i[A][2])))),P=d.Math.abs(n.oo(n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][1]),n.Rl(t,t.i[A][1])),n.no(t.J[A].a,t.J[A].b,n.Ql(t,t.i[A][2]),n.Rl(t,t.i[A][2])))),tt=true,e>r&&e>P?r+P<A1&&(tt=false,L=lt-.5*(n.Ql(t,t.i[A][0])+n.Ql(t,t.i[A][1])),c=ut-.5*(n.Rl(t,t.i[A][0])+n.Rl(t,t.i[A][1]))):r>e&&r>P?e+P<A1&&(tt=false,L=lt-.5*(n.Ql(t,t.i[A][0])+n.Ql(t,t.i[A][2])),c=ut-.5*(n.Rl(t,t.i[A][0])+n.Rl(t,t.i[A][2]))):e+r<A1&&(tt=false,L=lt-.5*(n.Ql(t,t.i[A][1])+n.Ql(t,t.i[A][2])),c=ut-.5*(n.Rl(t,t.i[A][1])+n.Rl(t,t.i[A][2]))),tt){for(z=t.i[A][0],l=s0,k=0;k<3;k++)i=t.i[A][k],i!=Qt&&(Q=d.Math.pow(t.J[A].a-t.J[i].a,2)+d.Math.pow(t.J[A].b-t.J[i].b,2),Q<l&&(z=i,l=Q,h.Pkb()));p=n.Tk(t,(t.J[Qt].a+t.J[z].a)/2,(t.J[Qt].b+t.J[z].b)/2,0);}else p=n.Tk(t,lt+L,ut+c,0);}else p=n.Tk(t,lt+L,ut+c,0);n.Cn(t,p,1),n.Wk(t,A,p,1);}break;case 2:if(K=t.j[A],K==1)L=lt-n.Ql(t,t.i[A][0]),c=ut-n.Rl(t,t.i[A][0]),p=n.Tk(t,lt+(Xt.Ez*L-Xt.Jz*c)*.7,ut+(Xt.Jz*L+Xt.Ez*c)*.7,0),n.Cn(t,p,1),n.Wk(t,A,p,1),p=n.Tk(t,lt+(Xt.Bz*L-Xt.Gz*c)*.7,ut+(Xt.Gz*L+Xt.Bz*c)*.7,0),n.Cn(t,p,1),n.Wk(t,A,p,1);else if(K==2)y=lt-n.Ql(t,t.i[A][0]),b=ut-n.Rl(t,t.i[A][0]),T=lt-n.Ql(t,t.i[A][1]),g=ut-n.Rl(t,t.i[A][1]),q=d.Math.sqrt(y*y+b*b)*.7,V=d.Math.sqrt(T*T+g*g)*.7,L=y+T,c=b+g,E=d.Math.sqrt(L*L+c*c),D=(q+V)/2,L=L/E*D,c=c/E*D,it=n.Ip(t,A),p=n.Tk(t,lt+Xt.Az*L-Xt.Fz*c,ut+Xt.Fz*L+Xt.Az*c,0),n.Cn(t,p,1),it>-1?n.Wk(t,A,p,1):n.Wk(t,A,p,257),p=n.Tk(t,lt+Xt.Cz*L-Xt.Hz*c,ut+Xt.Hz*L+Xt.Cz*c,0),n.Cn(t,p,1),n.Wk(t,A,p,1);else for(S=0;S<G;S++)p=n.Tk(t,lt,ut,0),n.Cn(t,p,1),n.Wk(t,A,p,1);break;case 3:t.i[A][0]>0?(L=(lt-n.Ql(t,t.i[A][0]))*.7,c=(ut-n.Rl(t,t.i[A][0]))*.7,p=n.Tk(t,lt+L,ut+c,0),n.Cn(t,p,1),n.Wk(t,A,p,1),p=n.Tk(t,lt-c,ut+L,0),n.Cn(t,p,1),n.Wk(t,A,p,1),p=n.Tk(t,lt+c,ut-L,0),n.Cn(t,p,1),n.Wk(t,A,p,1)):(L=n.Vo(t,true),c=n.Vo(t,true),p=n.Tk(t,lt+L,ut+c,0),n.Cn(t,p,1),n.Wk(t,A,p,1),p=n.Tk(t,lt-c*Xt.Ez,ut+L*Xt.Jz,0),n.Cn(t,p,1),n.Wk(t,A,p,1),p=n.Tk(t,lt-c*Xt.Ez,ut-L*Xt.Jz,0),n.Cn(t,p,1),n.Wk(t,A,p,1));break;default:{for(S=0;S<G;S++)p=n.Tk(t,lt,ut,0),n.Cn(t,p,1),n.Wk(t,A,p,1);break;}}},Xt.Nz=function(t){Xt.Kz();var A,e;for(n.bw(t,1),e=0,A=0;A<t.q;A++)t.C[A]==1?++e:e+=t.e[A]-t.j[A]+n.op(t,A);return e;},Xt.Az=0,Xt.Bz=0,Xt.Cz=0,Xt.Dz=0,Xt.Ez=0,Xt.Fz=0,Xt.Gz=0,Xt.Hz=0,Xt.Iz=0,Xt.Jz=0,X.Pz=function(t,A,e){var r,P,i;for(i=A.length,r=new X.uB(t.f,i,false),r.a[0]=0,r.b[0]=0,P=0;P<i;P++)r.k[P]=128-i,r.e[P]=A[P];i<8?X.Yz(r):X.Wz(t,r,A,e),X.FA(t.d,r);},X.Qz=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T;for(D=B(o.Q7,H,6,P,15,1),i=0;i<t.f.q;i++){for(Q=0;Q<n.Lo(t.f,i);Q++){for(T=e[n.Zo(t.f,i,Q)],l=0;l<Q&&!(T<D[l]);l++);for(L=Q;L>l;L--)D[L]=D[L-1];D[l]=T;}for(y=n.Lo(t.f,i),n.eh(A[i],i),n.ah(A[i],r,o.Kfb(e[i])),n.ah(A[i],(P-y)*(r+1),0),u=0;u<y;u++)n.ah(A[i],r+1,o.Kfb(D[u]));}},X.Rz=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(r=n.Pg(t.f.f),Q=2,e=0;e<t.f.f;e++)Q=d.Math.max(Q,n.Lo(t.f,e));for(i=(62+2*r+Q*(r+1))/63|0,P=B(o.a8,Hn,101,t.f.q,0,1),A=0;A<t.f.q;A++)P[A]=new n.fh(i),n.eh(P[A],A);for(y=B(o.Q7,H,6,t.f.q,15,1),D=0;D<t.f.g;D++)u=n.gm(t.f,D),(u==1||u==2)&&(n.bh(P[n.Zl(t.f,0,D)],o.Kfb(u)),n.bh(P[n.Zl(t.f,1,D)],o.Kfb(u)));l=X.Sz(P,y);do L=l,X.Qz(t,P,y,r,Q),l=X.Sz(P,y);while(L!=l);return y;},X.Sz=function(t,A){var e,r;for(r=0,U.Rrb(t,0,t.length,null),e=0;e<t.length;e++)(e==0||n.dh(t[e],t[e-1])!=0)&&++r,A[t[e].a]=r;return r;},X.Tz=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T;for(Q=new U.rrb(t.d);Q.a<Q.c.a.length;)for(u=U.qrb(Q),l=0;l<u.f.length;l++)if(r=u.f[l],n.fm(t.f,r)==2&&(!n.jq(t.f,r)&&(n.gm(t.f,r)==3||n.gm(t.f,r)==0)&&n.Kn(t.f,r),!n.gq(t.f,r)&&n.$o(t.f,n.Zl(t.f,0,r))>1&&n.$o(t.f,n.Zl(t.f,1,r))>1&&(n.gm(t.f,r)==1||n.gm(t.f,r)==2))){for(T=B(o.Q7,H,6,2,15,1),P=B(o.Q7,H,6,2,15,1),L=0;L<2;L++)for(T[L]=t.f.M,P[L]=n.Zl(t.f,L,r),y=0;y<n.Lo(t.f,P[L]);y++)i=n.Zo(t.f,P[L],y),i!=n.Zl(t.f,1-L,r)&&T[L]>i&&(T[L]=i);D=X.$A(u.a[u.g[P[0]]],u.b[u.g[P[0]]],u.a[u.g[P[1]]],u.b[u.g[P[1]]]),A=X.$A(u.a[u.g[T[0]]],u.b[u.g[T[0]]],u.a[u.g[P[0]]],u.b[u.g[P[0]]]),e=X.$A(u.a[u.g[P[1]]],u.b[u.g[P[1]]],u.a[u.g[T[1]]],u.b[u.g[T[1]]]),X.$z(D,A)<0^X.$z(D,e)<0^n.gm(t.f,r)==2&&X.iB(u,r);}},X.Uz=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;for(T=new U.Mj(),v=new U.Mj(),r=new U.Mj(),i=new U.rrb(t.d);i.a<i.c.a.length;){for(P=U.qrb(i),D=0,w.kAb(r.a,0),u=0;u<P.e.length;u++)A=P.e[u],e=t.i[A],e!=0&&(U.sj(r,new X.aB(P,u,e)),D+=e);if(D!=0)for(U.Jj(r,new X.RA()),l=new U.rrb(r);l.a<l.c.a.length&&(Q=U.qrb(l),!(D*Q.b>0&&(e=d.Math.abs(D)>=d.Math.abs(Q.b)?Q.b:D,D-=e,U.sj(e<0?T:v,new X.aB(P,Q.a,e)),D==0))););}if(T.a.length==0||v.a.length==0)return null;for(U.Jj(v,new X.KA()),U.Jj(T,new X.OA()),g=new U.rrb(v);g.a<g.c.a.length;)for(c=U.qrb(g),y=new U.rrb(T);y.a<y.c.a.length;)if(L=U.qrb(y),c.b==-L.b)return t.i[X.mB(c.c,c.a)]-=c.b,t.i[X.mB(L.c,L.a)]-=L.b,new X.XA(c.c,L.c,c.a,L.a);for(k=new U.rrb(v);k.a<k.c.a.length;)for(c=U.qrb(k),y=new U.rrb(T);y.a<y.c.a.length;)if(L=U.qrb(y),c.b>-L.b)return t.i[X.mB(c.c,c.a)]+=L.b,t.i[X.mB(L.c,L.a)]-=L.b,new X.XA(c.c,L.c,c.a,L.a);for(b=new U.rrb(v);b.a<b.c.a.length;)for(c=U.qrb(b),y=new U.rrb(T);y.a<y.c.a.length;)if(L=U.qrb(y),c.b<-L.b)return t.i[X.mB(c.c,c.a)]-=c.b,t.i[X.mB(L.c,L.a)]+=c.b,new X.XA(c.c,L.c,c.a,L.a);return null;},X.Vz=function(t){return t.d.a.length<2?null:new X.WA(U.xj(t.d,0),U.xj(t.d,1));},X.Wz=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O;if(L=j(F(o.O7,2),o1,15,0,[j(F(o.O7,1),Ft,6,15,[20]),null,null,j(F(o.O7,1),Ft,6,15,[0,10]),null,null,j(F(o.O7,1),Ft,6,15,[-4,12]),j(F(o.O7,1),Ft,6,15,[0,0,-7.5]),null,null,null,null,j(F(o.O7,1),Ft,6,15,[8.571428571428571,-8.571428571428571]),null,null,null,j(F(o.O7,1),Ft,6,15,[-2.4])]),y=j(F(o.Q7,2),E1,7,0,[j(F(o.Q7,1),H,6,15,[146]),j(F(o.Q7,1),H,6,15,[627]),null,j(F(o.Q7,1),H,6,15,[2457,1170]),null,j(F(o.Q7,1),H,6,15,[2451,8643,2519]),j(F(o.Q7,1),H,6,15,[9362,14798]),j(F(o.Q7,1),H,6,15,[34377,-2147448999,26214]),null,j(F(o.Q7,1),H,6,15,[37449,137313,95703,34371,37815,54891,132867,-2147309741,54857,55129,-2147449005,-2147449065]),null,j(F(o.Q7,1),H,6,15,[530697,531819,899169,137289,694617,-2146951863,-2146952797,-2146939175,-2146929547,-2146929564,-2146625111,-2146931799,-2146940503,-2146931935]),j(F(o.Q7,1),H,6,15,[1007293,610915]),j(F(o.Q7,1),H,6,15,[542985,137283,2122017,530691,2206773,-2144711351,219209,2840841,137555,-2146871031,-2147264167,613705,-2145360543,-2146625271,694611,2454837,-2145356703,-2147345133,-2146928951,-2146931805,-2144641719,-2146951869,-2146625237,-2146624183,2841963,1074905,-2146625117,2799955,-2144723645,138583,859225,-2145264843,-2145216253,-2146624149,-2144700727,-2146928917,-2143905527,-2144045771,-2146789097,2288547,544407,2104323,-2146911977,-2144479405,3633737,-2146870089,-2146952169]),null,j(F(o.Q7,1),H,6,15,[8487297,2172633,2116611,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8829813]),j(F(o.Q7,1),H,6,15,[14071213])]),S=1<<A.e.length,D=0,Q=0,A.e.length>7)for(c=0;c<A.e.length;c++)u=X.cA(t,e,r,c),u==1?D+=S:u==2&&(Q+=S),D>>>=1,Q>>>=1;if(q=A.e.length-9,A.e.length>=9&&A.e.length<=25&&y[q]!=null)for(O=0;O<y[q].length;O++)for(v=(r1&y[q][O])==0,l=qt&y[q][O],g=false;!g;g=!g){if(g){if(v)break;for(p=0,P=1;P!=S;P<<=1)p<<=1,(l&P)!=0&&(p|=1);l=p;}for(V=0;V<A.e.length;V++){if((l&D)==0&&(~l&Q)==0){for(i=0,T=e9*(L[q]==null?0:L[q][O]),E=0,G=l,k=true,b=0;b<A.e.length;b++)k&&++E,(G&1)==0&&(k=!k),G>>>=1;for(K=E>(A.e.length/2|0),c=1;c<A.e.length;c++)A.a[c]=A.a[c-1]+d.Math.sin(i),A.b[c]=A.b[c-1]+d.Math.cos(i),(l&1)==0&&(K=!K),i+=T+(K?c0:be),l>>>=1;return;}(l&1)!=0&&(l|=S),l>>>=1;}}X.Zz(A,D,Q);},X.Xz=function(t){var A,e,r,P,i,D,u,Q,l,L;for(A=null,l=null,D=0;D<t.f.g;D++)if(n.im(t.f,D)==16){for(e=n.Zl(t.f,0,D),P=-1,u=0;u<t.d.a.length&&(P=X.nB(U.xj(t.d,u),e),P==-1);u++);for(r=n.Zl(t.f,1,D),i=-1,Q=0;Q<t.d.a.length&&(i=X.nB(U.xj(t.d,Q),r),i==-1);Q++);u!=Q&&(u>Q&&(L=u,u=Q,Q=L,L=P,P=i,i=L),l==null&&(l=B(o.h9,b0,461,t.d.a.length,0,2)),l[Q]==null&&(l[Q]=B(o.h9,{461:1,4:1,5:1,8:1},106,Q,0,1)),l[Q][u]?X.UA(l[Q][u],P,i):(a1(l[Q],u,new X.XA(U.xj(t.d,u),U.xj(t.d,Q),P,i)),!A&&(A=new U.Mj()),U.sj(A,l[Q][u])));}return A;},X.Yz=function(t){var A,e;for(A=A1-A1*(t.e.length-2)/t.e.length,e=1;e<t.e.length;e++)t.a[e]=t.a[e-1]+d.Math.sin(A*(e-1)),t.b[e]=t.b[e-1]+d.Math.cos(A*(e-1));},X.Zz=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;if(A==0||(A&e)!=0){X.Yz(t);return;}for(k=-1,v=0,i=1<<t.e.length-2,P=1<<t.e.length-1,u=1,D=2,l=0;l<t.e.length;l++)(e&(P|u))==0&&(A&(P|u))!=0&&(A&i)==0&&(b=0,(e&i)!=0&&(b+=4),(A&P)!=0&&(b+=2),(A&u)!=0&&(b+=1),v<b&&(v=b,k=l)),i=P,P=u,u=D,D=1<<(l+2<t.e.length?l+2:l+2-t.e.length);if(k==-1){X.Yz(t);return;}for(T=0,T|=1<<k,c=2;c<t.e.length-1;){if(y=k+c<t.e.length?k+c:k+c-t.e.length,P=1<<(y==0?t.e.length-1:y-1),(e&P)!=0){++c;continue;}if(u=1<<y,(A&P)!=0){if((e&u)!=0){X.Yz(t);return;}T|=u,c+=2;continue;}if(D=1<<(y+1<t.e.length?y+1:y+1-t.e.length),(A&u)!=0&&(e&D)!=0){T|=u,c+=3;continue;}++c;}if(T==0){X.Yz(t);return;}for(r=A1-A1*(t.e.length-2)/t.e.length,L=1;L<t.e.length;L++)t.a[L]=t.a[L-1]+d.Math.sin(r*(L-1)),t.b[L]=t.b[L-1]+d.Math.cos(r*(L-1));for(u=1,g=2*d.Math.sin(r/2),Q=0;Q<t.e.length;Q++)(T&u)!=0&&(t.a[Q]+=g*d.Math.cos(r*(Q-.5)),t.b[Q]-=g*d.Math.sin(r*(Q-.5))),u<<=1;},X.$z=function(t,A){var e;for(e=t-A;e<oP;)e+=N1;for(;e>A1;)e-=N1;return e;},X._z=function(t,A,e){var r,P;for(r=0,P=0;P<n.Lo(t.f,e);P++)X.pB(A,n.Zo(t.f,e,P))&&++r;return r;},X.aA=function(t,A,e,r){var P,i,D,u,Q;return u=X.nB(A,r),Q=X.nB(e,r),X.tB(e,A.a[u]-e.a[Q],A.b[u]-e.b[Q]),P=X.zA(t,A,r),i=X.zA(t,e,r),D=0,X._z(t,A,r)==1&&X._z(t,e,r)==1&&(D=c0),X.sB(e,e.a[Q],e.b[Q],P-i+D+A1),X.fA(t,A,e,1);},X.bA=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot;for(p=B(o.Q7,H,6,P,15,1),E=B(o.Q7,H,6,P,15,1),g=0;g<P;g++)p[g]=X.nB(A,r[g]),E[g]=X.nB(e,r[g]);for(Qt=0,it=0,z=0,lt=0,k=0;k<P;k++)Qt+=A.a[p[k]],it+=A.b[p[k]],z+=e.a[E[k]],lt+=e.b[E[k]];for(Qt/=P,it/=P,z/=P,lt/=P,X.tB(e,Qt-z,it-lt),l=B(o.i9,R1,41,P,0,1),y=B(o.i9,R1,41,P,0,1),i=B(o.i9,R1,41,P,0,1),D=B(o.i9,R1,41,P,0,1),v=0;v<P;v++)l[v]=new X.ZA(Qt,it,A.a[p[v]],A.b[p[v]]),y[v]=new X.ZA(Qt,it,e.a[E[v]],e.b[E[v]]),i[v]=new X.YA(l[v].a-y[v].a,l[v].b*y[v].b),D[v]=new X.YA(l[v].a+y[v].a,l[v].b*y[v].b);for(V=X.DA(i,P),G=X.DA(D,P),Dt=0,ot=0,S=0;S<P;S++)for(q=0;q<n.Lo(t.f,r[S]);q++)u=n.Zo(t.f,r[S],q),X.pB(A,u)&&!X.pB(e,u)&&++Dt,!X.pB(A,u)&&X.pB(e,u)&&++ot;for(L=B(o.i9,R1,41,Dt,0,1),T=B(o.i9,R1,41,ot,0,1),c=B(o.i9,R1,41,ot,0,1),Dt=0,ot=0,b=0;b<P;b++)for(q=0;q<n.Lo(t.f,r[b]);q++)u=n.Zo(t.f,r[b],q),X.pB(A,u)&&!X.pB(e,u)&&(Q=X.nB(A,u),L[Dt]=new X.ZA(A.a[p[b]],A.b[p[b]],A.a[Q],A.b[Q]),++Dt),!X.pB(A,u)&&X.pB(e,u)&&(Q=X.nB(e,u),ut=new X.ZA(e.a[E[b]],e.b[E[b]],e.a[Q],e.b[Q]),T[ot]=new X.YA(V.a+ut.a,ut.b),c[ot]=new X.YA(G.a-ut.a,ut.b),++ot);return K=X.DA(L,Dt),O=X.DA(T,ot),tt=X.DA(c,ot),d.Math.abs(X.$z(K.a,O.a))>d.Math.abs(X.$z(K.a,tt.a))?X.sB(e,Qt,it,V.a):(X.hB(e,Qt,it),X.sB(e,Qt,it,G.a)),X.fA(t,A,e,P);},X.cA=function(t,A,e,r){var P,i,D,u,Q,l;return i=r==A.length-1?0:r+1,u=r==0?A.length-1:r-1,D=i==A.length-1?0:i+1,n.fm(t.f,e[r])==2&&(P=n.gm(t.f,e[r]),P==1||P==2)?(X.kA(t,A[u],A[r],A[i])^X.kA(t,A[D],A[i],A[r])&&(P=P==1?2:1),P):n.jq(t.f,e[r])?(Q=n.lt(n.Ep(t.f),e[u],e[r]),l=n.lt(n.Ep(t.f),e[i],e[r]),Q!=-1||l!=-1?Q==l?2:1:2):0;},X.dA=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c;for(P=B(o.Q7,H,6,t.f.q,15,1),i=B(o.Q7,H,6,t.f.q,15,1),D=B(o.Q7,H,6,t.f.q,15,1),u=B(o.Q7,H,6,t.f.q,15,1),P[0]=A,D[A]=1,u[0]=-1,r=0,Q=0;r<=Q;){if(r==0||!t.b[P[r]])for(l=0;l<n.Lo(t.f,P[r]);l++)e=n.Zo(t.f,P[r],l),T=n._o(t.f,P[r],l),D[e]==0&&!t.c[T]&&(P[++Q]=e,i[Q]=T,D[e]=D[P[r]]+1,u[Q]=r);if(r==Q){for(c=new X._A(D[P[r]]),L=r,y=0;y<c.a.length;y++)c.a[y]=P[L],c.b[y]=i[L],L=u[L];return c;}++r;}return null;},X.eA=function(t){var A,e,r,P;for(P=0,r=null,e=new U.rrb(t);e.a<e.c.a.length;)A=U.qrb(e),P<A.b[0].e.length*A.b[1].e.length&&(P=A.b[0].e.length*A.b[1].e.length,r=A);return r;},X.fA=function(t,A,e,r){var P,i,D,u,Q;for(i=new X.uB(t.f,A.e.length+e.e.length-r,A.i|e.i),P=0,u=0;u<A.e.length;u++)i.e[P]=A.e[u],i.k[P]=A.k[u],i.a[P]=A.a[u],i.b[P++]=A.b[u];for(D=0;D<e.e.length;D++)Q=X.nB(A,e.e[D]),Q==-1?(i.e[P]=e.e[D],i.k[P]=e.k[D],i.a[P]=e.a[D],i.b[P++]=e.b[D]):i.k[Q]<e.k[D]&&(i.k[Q]=e.k[D],i.a[Q]=e.a[D],i.b[Q]=e.b[D]);return i;},X.gA=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;for(u=B(o.Q7,H,6,t.f.q,15,1),Q=B(o.Q7,H,6,t.f.q,15,1),l=B(o.Q7,H,6,t.f.q,15,1),L=B(o.Q7,H,6,t.f.q,15,1),u[0]=e,l[e]=1,L[0]=-1,D=0,y=0;D<=y;){for(T=0;T<n.Lo(t.f,u[D]);T++){if(P=n.Zo(t.f,u[D],T),b=n._o(t.f,u[D],T),P==A){for(i=l[u[D]],r=B(o.Q7,H,6,i,15,1),r[0]=b,c=1;c<i;c++)r[c]=Q[D],D=L[D];return r;}l[P]==0&&(u[++y]=P,Q[y]=b,l[P]=l[u[D]]+1,L[y]=D);}if(D==y)return null;++D;}return null;},X.hA=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b;for(e=n.Zl(t.f,0,A),r=n.Zl(t.f,1,A),D=B(o.Q7,H,6,t.f.q,15,1),u=B(o.Q7,H,6,t.f.q,15,1),Q=B(o.Q7,H,6,t.f.q,15,1),l=B(o.Q7,H,6,t.f.q,15,1),D[0]=e,D[1]=r,u[1]=A,Q[e]=1,Q[r]=2,l[0]=-1,l[1]=0,i=1,L=1;i<=L;){for(y=0;y<n.$o(t.f,D[i]);y++){if(P=n.Zo(t.f,D[i],y),i>1&&P==e){for(b=new X._A(Q[D[i]]),u[0]=n._o(t.f,D[i],y),T=i,c=0;c<b.a.length;c++)b.a[c]=D[T],b.b[c]=u[T],T=l[T];return b;}Q[P]==0&&n.fq(t.f,P)&&(D[++L]=P,u[L]=n._o(t.f,D[i],y),Q[P]=Q[D[i]]+1,l[L]=i);}++i;}return null;},X.iA=function(t,A,e,r){var P,i,D,u,Q,l;for(D=B(o.Q7,H,6,t.f.q,15,1),u=B(o.Q7,H,6,t.f.q,15,1),D[0]=e,D[1]=A,u[e]=1,u[A]=2,i=1,Q=1;i<=Q;){for(l=0;l<n.$o(t.f,D[i]);l++){if(P=n.Zo(t.f,D[i],l),P==r)return 1+u[D[i]];u[P]==0&&n.fq(t.f,P)&&(D[++Q]=P,u[P]=u[D[i]]+1);}++i;}return 0;},X.jA=function(t,A){var e,r,P,i,D,u,Q;for(u=(A.T&15)!=0,Q=A.T&248,!t.g&&(t.g=new U.twb()),(t.e&2)!=0&&n.qq(A,false),t.f=A,n.bw(t.f,7),t.d=new X.HA(),t.b=B(o.vfb,Tt,6,t.f.q,16,1),t.c=B(o.vfb,Tt,6,t.f.r,16,1),t.i=B(o.Q7,H,6,t.f.q,15,1),e=0;e<t.f.q;e++)t.i[e]=n.Bl(t.f,e);for((t.e&12)!=0&&X.uA(t),(t.e&1)==0&&X.Oz&&X.wA(t,X.Oz),X.tA(t),X.pA(t),X.rA(t),X.pA(t),i=new U.rrb(t.d);i.a<i.c.a.length;)r=U.qrb(i),X.qB(r);for(X.Tz(t),X.xA(t),X.vA(t),X.oA(t),X.mA(t),X.qA(t),P=new U.rrb(t.d);P.a<P.c.a.length;)for(r=U.qrb(P),D=0;D<r.e.length;D++)n.zn(t.f,r.e[D],r.a[D]),n.An(t.f,r.e[D],r.b[D]),n.Bn(t.f,r.e[D],0);u&&(n.vq(t.f,Q),n.yq(t.f)),t.a&&n.Zm(t.f);},X.kA=function(t,A,e,r){var P,i;for(i=0;i<n.$o(t.f,e);i++)if(P=n.Zo(t.f,e,i),P!=r&&P<A)return  false;return  true;},X.lA=function(t,A,e){var r;X.VA(A,e,(t.e&12)!=0),r=X.fA(t,A.b[0],A.b[1],0),X.AA(t,A.b[0],A.b[1],r);},X.mA=function(t){var A;for(A=X.Uz(t);A;)X.lA(t,A,Ue),A=X.Uz(t);},X.nA=function(t,A,e,r){var P,i,D,u,Q;for(P=B(o.Q7,H,6,r,15,1),i=0,D=0;D<A.e.length;D++)for(u=0;u<e.e.length;u++)A.e[D]==e.e[u]&&(P[i++]=A.e[D]);Q=r==1?X.aA(t,A,e,P[0]):X.bA(t,A,e,P,r),X.AA(t,A,e,Q);},X.oA=function(t){var A,e;for(e=X.Xz(t);e;)A=X.eA(e),X.lA(t,A,Te),e=X.Xz(t);},X.pA=function(t){for(var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;;){for(S=0,c=0,k=null,v=null,D=1;D<t.d.a.length;D++)for(r=U.xj(t.d,D),u=0;u<D;u++){for(P=U.xj(t.d,u),A=0,e=0,b=0,g=0,L=0;L<r.e.length;L++)for(T=0;T<P.e.length;T++)r.e[L]==P.e[T]&&(++e,A=r.e[L],b<r.k[L]&&(b=r.k[L]),g<P.k[T]&&(g=P.k[T]));if(e>0&&(i=e==1&&X._z(t,r,A)==1&&X._z(t,P,A)==1?0:1,b>g?Q=(i<<24)+(b<<16)+(g<<8)+e:Q=(i<<24)+(g<<16)+(b<<8)+e,S<Q)){for(S=Q,c=e,b=0,g=0,y=0;y<r.e.length;y++)b<r.k[y]&&(b=r.k[y]);for(l=0;l<P.e.length;l++)g<P.k[l]&&(g=P.k[l]);b>g?(k=r,v=P):(k=P,v=r);}}if(S==0)break;c==k.e.length?U.Fj(t.d,k):c==v.e.length?U.Fj(t.d,v):X.nA(t,k,v,c);}},X.qA=function(t){var A;for(A=X.Vz(t);A;)X.lA(t,A,Ci),A=X.Vz(t);},X.rA=function(t){for(var A,e,r,P,i,D,u;;){for(i=null,A=0;A<t.f.q;A++){for(u=0,P=0;P<n.Lo(t.f,A);P++)t.c[n._o(t.f,A,P)]||++u;u==1&&(D=X.dA(t,A),(!i||D.a.length>i.a.length)&&(i=D));}if(!i)break;for(e=new X.uB(t.f,i.a.length,false),r=0;r<i.a.length;r++)t.b[i.a[r]]=true,r<i.a.length-1&&(t.c[i.b[r]]=true),e.e[r]=i.a[r],e.a[r]=d.Math.cos(Sn)*r,e.b[r]=(r&1)==1?0:.5,e.k[r]=128+i.a.length;X.FA(t.d,e);}},X.sA=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T;for(i=0;i<t.f.r;i++)if(r=n.Zl(t.f,0,i),P=n.Zl(t.f,1,i),!(n.gq(t.f,i)||n.fm(t.f,i)!=1||n.Lo(t.f,r)==1||n.Lo(t.f,P)==1)&&!((t.e&4)!=0&&n.Km(t.f,r)&&n.Km(t.f,P))){for(y=false,l=0;l<2;l++)if(D=n.Zl(t.f,l,i),n.Lo(t.f,D)>2){for(T=true,Q=-1,L=0;L<n.Lo(t.f,D);L++)u=n.Zo(t.f,D,L),u!=n.Zl(t.f,1-l,i)&&(Q==-1?Q=e[u]:Q!=e[u]&&(T=false));if(T){y=true;break;}}y||((t.e&8)!=0&&n.Km(t.f,r)&&n.Km(t.f,P)?A[i]=1:A[i]=2);}},X.tA=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt;for(r=0;r<t.f.f;r++)if(n.Lo(t.f,r)>4){for(T=new X.uB(t.f,1+n.Lo(t.f,r),false),T.a[n.Lo(t.f,r)]=0,T.b[n.Lo(t.f,r)]=0,T.k[n.Lo(t.f,r)]=32,T.e[n.Lo(t.f,r)]=r,t.b[r]=true,b=0;b<n.Lo(t.f,r);b++)l=n.Zo(t.f,r,b),T.a[b]=d.Math.sin(c0*b-In),T.b[b]=d.Math.cos(c0*b-In),T.k[b]=32,T.e[b]=l,t.b[l]=true,t.c[n._o(t.f,r,b)]=true;X.FA(t.d,T);}for(lt=n.Ep(t.f),it=0;it<lt.j.a.length;it++){if(ut=U.xj(lt.k,it).length,Qt=U.xj(lt.j,it),Dt=false,(t.e&12)!=0){for(Dt=true,b=0;b<ut;b++)if(!n.Km(t.f,Qt[b])){Dt=false;break;}}if(!Dt){for(v=false,g=0;g<ut;g++)if(n.Uo(t.f,Qt[g])==ut){v=true;break;}if(v)for(z=U.xj(lt.k,it),X.Pz(t,Qt,z),b=0;b<ut;b++)t.b[Qt[b]]=true,t.c[z[b]]=true;}}for(u=0;u<t.f.g;u++)if(n.gq(t.f,u)&&!t.c[u])for(bt=X.hA(t,u),Qt=bt.a,z=bt.b,X.Pz(t,Qt,z),b=0;b<bt.a.length;b++)t.b[Qt[b]]=true,t.c[z[b]]=true;for(Q=0;Q<t.f.r;Q++)if(!t.c[Q]&&n.fm(t.f,Q)==3&&(P=n.Zl(t.f,0,Q),i=n.Zl(t.f,1,Q),V=n.Lo(t.f,P)+n.Lo(t.f,i),V>2)){for(T=new X.uB(t.f,V,false),L=0,g=0;g<n.Lo(t.f,P);g++)l=n.Zo(t.f,P,g),l!=i&&(T.e[L++]=l,t.b[l]=true,t.c[n._o(t.f,P,g)]=true);for(T.e[L++]=P,T.e[L++]=i,k=0;k<n.Lo(t.f,i);k++)l=n.Zo(t.f,i,k),l!=P&&(T.e[L++]=l,t.b[l]=true,t.c[n._o(t.f,i,k)]=true);for(b=0;b<V;b++)T.a[b]=b,T.b[b]=0,T.k[b]=1;t.b[P]=true,t.b[i]=true,t.c[Q]=true,X.FA(t.d,T);}for(D=0;D<t.f.r;D++)if(!t.c[D]&&n.fm(t.f,D)==2){for(A=B(o.Q7,H,6,t.f.q,15,1),b=0;b<2;b++)if(A[0]=n.Zl(t.f,b,D),A[1]=n.Zl(t.f,1-b,D),n.Qo(t.f,A[0])==1&&n.Qo(t.f,A[1])==2&&n.Lo(t.f,A[1])==2){t.b[A[0]]=true,t.b[A[1]]=true,t.c[D]=true,q=1;do{if(G=n.Zo(t.f,A[q],0)==A[q-1]?1:0,A[q+1]=n.Zo(t.f,A[q],G),n.Qo(t.f,A[q+1])==2&&n.Lo(t.f,A[q+1])>2)break;t.b[A[q+1]]=true,t.c[n._o(t.f,A[q],G)]=true,++q;}while(n.Qo(t.f,A[q])==2&&n.Lo(t.f,A[q])==2);for(V=n.Lo(t.f,A[0])+n.Lo(t.f,A[q])+q-1,T=new X.uB(t.f,V,false),p=0;p<=q;p++)T.a[p]=p,T.b[p]=0,T.k[p]=64,T.e[p]=A[p];for(y=q+1,c=false,E=0;E<n.Lo(t.f,A[0]);E++)l=n.Zo(t.f,A[0],E),l!=A[1]&&(T.a[y]=-0.5,T.b[y]=c?d.Math.sin(c0):-d.Math.sin(c0),T.k[y]=64,T.e[y]=l,++y,c=true);for(c=false,S=0;S<n.Lo(t.f,A[q]);S++)l=n.Zo(t.f,A[q],S),l!=A[q-1]&&(T.a[y]=q+.5,T.b[y]=c?-d.Math.sin(c0):d.Math.sin(c0),T.k[y]=64,T.e[y]=l,++y,c=true);X.FA(t.d,T);}}for(e=0;e<t.f.q;e++)if(n.Lo(t.f,e)==4){for(K=B(o.Q7,H,6,4,15,1),O=B(o.Q7,H,6,4,15,1),tt=0,g=0;g<4;g++)K[tt]=n.Zo(t.f,e,g),O[tt]=n._o(t.f,e,g),n.Lo(t.f,K[tt])==1&&!t.c[O[tt]]&&++tt;if(tt==2){for(T=new X.uB(t.f,3,false),b=0;b<2;b++)t.b[K[b]]=true,t.c[O[b]]=true,T.e[b]=K[b],T.k[b]=32;T.a[0]=-0.5,T.b[0]=.866,T.a[1]=.5,T.b[1]=.866,T.a[2]=0,T.b[2]=0,T.k[2]=32,T.e[2]=e,X.FA(t.d,T);}if(tt==3){for(k=0;k<2;k++)n.fm(t.f,O[k])==1&&(ot=K[k],K[k]=K[2],K[2]=ot,ot=O[k],O[k]=O[2],O[2]=ot);for(T=new X.uB(t.f,4,false),b=0;b<3;b++)t.b[K[b]]=true,t.c[O[b]]=true,T.e[b]=K[b],T.k[b]=32;T.a[0]=-1,T.b[0]=0,T.a[1]=1,T.b[1]=0,T.a[2]=0,T.b[2]=1,T.a[3]=0,T.b[3]=0,T.k[3]=32,T.e[3]=e,X.FA(t.d,T);}}},X.uA=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;for(u=0,e=0;e<t.f.q;e++)n.Km(t.f,e)&&++u;if(!(u<2)){for(y=0,l=0,L=0;L<t.f.r;L++)r=n.Zl(t.f,0,L),P=n.Zl(t.f,1,L),n.Km(t.f,r)&&n.Km(t.f,P)&&(t.c[L]=true,t.b[r]=true,t.b[P]=true,l+=n.dm(t.f,L),++y);for(y!=0&&l!=0?l/=y:l=n.Ul(t.f),i=0;i<t.f.q;i++)n.Km(t.f,i)&&!t.b[i]&&--u;if(!(u<2)){for(p=B(o.Q7,H,6,t.f.q,15,1),T=n.hp(t.f,p,true,true),S=B(o.Q7,H,6,T,15,1),D=0;D<t.f.q;D++)p[D]!=-1&&++S[p[D]];for(v=B(o.m9,R1,47,T,0,1),b=0;b<T;b++)v[b]=new X.uB(t.f,S[b],true);for(Q=B(o.Q7,H,6,T,15,1),A=0;A<t.f.q;A++)g=p[A],g!=-1&&(v[g].k[Q[g]]=f0,v[g].e[Q[g]]=A,v[g].a[Q[g]]=n.Ql(t.f,A)/l,v[g].b[Q[g]]=n.Rl(t.f,A)/l,++Q[g]);for(E=-1,q=0,k=0;k<T;k++)q<S[k]&&(q=S[k],E=k);for(X.FA(t.d,v[E]),c=0;c<T;c++)c!=E&&X.FA(t.d,v[c]);}}},X.vA=function(t){var A,e;for(A=0;A<t.f.q;A++)!t.b[A]&&n.Lo(t.f,A)==0&&(e=new X.uB(t.f,1,false),t.b[A]=true,e.e[0]=A,e.a[0]=0,e.b[0]=0,e.k[0]=0,X.FA(t.d,e));},X.wA=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;for(b=null,b=new n.iu(),n.cu(b,t.f),D=B(o.vfb,Tt,6,t.f.f,16,1),k=A.Tb();k.ed();)if(g=k.fd(),c=null,S=g.b,n._t(b,S),n.Pt(b,4,8)!=0&&(c=b.F),c)for(T=new U.rrb(c);T.a<T.c.a.length;){for(y=U.qrb(T),v=0,r=y,P=0,i=r.length;P<i;++P)e=r[P],D[e]&&++v;if(v<=1){for(Q=g.c,t.a?Q=false:t.a=true,l=new X.uB(t.f,y.length,Q),L=0;L<y.length;L++)e=y[L],Q&&n.sn(t.f,e,true),l.k[L]=256,l.e[L]=e,l.a[L]=n.Ql(g.b,L)/g.a,l.b[L]=n.Rl(g.b,L)/g.a,D[e]=true,t.b[e]=true;for(u=0;u<S.g;u++)t.c[n.Xo(t.f,y[S.D[0][u]],y[S.D[1][u]])]=true;X.FA(t.d,l);}}return D;},X.xA=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E;for(A=X.Rz(t),i=B(o.M7,Y,6,t.f.r,15,1),X.sA(t,i,A),P=0;P<t.f.r;P++)i[P]==2&&(n.fq(t.f,n.Zl(t.f,0,P))||n.fq(t.f,n.Zl(t.f,1,P)))&&(i[P]=3);for(c=0;c<t.d.a.length;c++){for(y=U.xj(t.d,c),Q=X.lB(y),v=y.c,k=new X.vB(y),g=-1,T=0;T<224&&Q.a.length!=0;T++){if(l=U.qwb(t.g,Q.a.length),u=(w.FAb(l,Q.a.length),Q.a[l]),D=X.gA(t,u[0],u[1]),e=B(o.Q7,H,6,D.length,15,1),r=0,T<32)for(b=1;b<D.length-1;b++)i[D[b]]==3&&(e[r++]=D[b]);else if(T<96)for(b=1;b<D.length-1;b++)i[D[b]]>=2&&(e[r++]=D[b]);else for(b=1;b<D.length-1;b++)i[D[b]]>=1&&(e[r++]=D[b]);if(r!=0){if(p=e[0],r>1)do p=e[U.qwb(t.g,r)];while(p==g);p!=g&&(g=p,X.iB(y,p),Q=X.lB(y),v>y.c&&(v=y.c,k=new X.vB(y)));}}U.Hj(t.d,c,k),y=k,L=1;do{for(S=9999,b=0;b<y.e.length;b++)E=A[y.e[b]],E==L?X.rB(y,b):E>L&&E<S&&(S=E);L=S;}while(S!=9999);}},X.yA=function(t,A){t.g=new U.uwb(A);},X.zA=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p;for(P=B(o.O7,Ft,6,n.Lo(t.f,e)+1,15,1),D=B(o.Q7,H,6,n.Lo(t.f,e)+1,15,1),u=B(o.Q7,H,6,n.Lo(t.f,e)+1,15,1),k=X.nB(A,e),i=0,l=0;l<n.Lo(t.f,e);l++)D[i]=n.Zo(t.f,e,l),u[i]=n._o(t.f,e,l),y=X.nB(A,D[i]),y!=-1&&(P[i++]=X.$A(A.a[k],A.b[k],A.a[y],A.b[y]));if(i==1)return P[0]+A1;for(L=i-1;L>0;L--)for(T=0;T<L;T++)P[T]>P[T+1]&&(v=P[T],P[T]=P[T+1],P[T+1]=v,S=D[T],D[T]=D[T+1],D[T+1]=S,p=u[T],u[T]=u[T+1],u[T+1]=p);for(P[i]=P[0]+N1,D[i]=D[0],u[i]=u[0],c=-100,b=0,Q=0;Q<i;Q++)r=P[Q+1]-P[Q],i>2&&n.gq(t.f,u[Q])&&n.gq(t.f,u[Q+1])&&(g=X.iA(t,D[Q],e,D[Q+1]),g!=0&&(r-=100-g)),c<r&&(c=r,b=Q);return (P[b]+P[b+1])/2;},X.AA=function(t,A,e,r){var P;P=d.Math.min(U.zj(t.d,A,0),U.zj(t.d,e,0)),U.rj(t.d,P,r),U.Fj(t.d,A),U.Fj(t.d,e);},X.BA=function(){X.CA.call(this,2);},X.CA=function(t){this.e=t,(t&1)==0&&!X.Oz&&!X.Oz&&(X.Oz=new X.dB());},X.DA=function(t,A){var e,r,P,i,D;for(D=0,e=0,r=0;r<A;r++)D+=t[r].b*d.Math.sin(t[r].a),e+=t[r].b*d.Math.cos(t[r].a);return e==0?i=D>0?v0:ue:(i=d.Math.atan(D/e),e<0&&(i+=A1)),P=d.Math.sqrt(D*D+e*e)/A,new X.YA(i,P);},X.EA=function(t,A){var e,r;return e=d.Math.abs(t.b),r=d.Math.abs(A.b),e<r?-1:e>r?1:0;},x(68,1,{},X.BA,X.CA),s.a=false,s.e=0,o.g9=N(68),X.FA=function(t,A){var e,r;for(r=new U.rrb(t);r.a<r.c.a.length;)if(e=U.qrb(r),X.gB(e,A))return  false;return w.iAb(t.a,A),true;},X.GA=function(t,A){return X.FA(t,A);},X.HA=function(){U.Mj.call(this);},x(315,20,We,X.HA),s.add=function(t){return X.GA(this,t);},o.c9=N(315),X.JA=function(t){var A,e;return A=t.c.e.length,e=t.c.e.length,e<A?-1:e>A?1:0;},X.KA=function(){},x(317,1,{},X.KA),s.Rb=function(t,A){var e;return X.JA((e=t,e));},s.mb=function(t){return this===t;},o.d9=N(317),X.NA=function(t){var A,e;return A=t.c.e.length,e=t.c.e.length,A<e?-1:A>e?1:0;},X.OA=function(){},x(318,1,{},X.OA),s.Rb=function(t,A){var e;return X.NA((e=t,e));},s.mb=function(t){return this===t;},o.e9=N(318),X.RA=function(){},x(316,1,{},X.RA),s.Rb=function(t,A){return X.EA(t,A);},s.mb=function(t){return this===t;},o.f9=N(316),X.UA=function(t,A,e){t.c[0]+=X.jB(t.b[0],A),t.d[0]+=X.kB(t.b[0],A),t.c[1]+=X.jB(t.b[1],e),t.d[1]+=X.kB(t.b[1],e),++t.a[0],++t.a[1];},X.VA=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V;for(r=B(o.O7,Ft,6,2,15,1),l=0;l<2;l++)t.c[l]/=t.a[l],t.d[l]/=t.a[l],r[l]=X.eB(t.b[l],t.c[l],t.d[l],t.b[1-l].e.length,A);for(X.sB(t.b[0],t.c[0],t.d[0],v0-r[0]),X.sB(t.b[1],t.c[1],t.d[1],4.71238898038469-r[1]),V=s0,q=QP,D=t.d[0]-t.d[1],L=0;L<t.b[1].b.length;L++)t.b[1].b[L]+=D,V>t.b[1].b[L]&&(V=t.b[1].b[L]),q<t.b[1].b[L]&&(q=t.b[1].b[L]);for(p=q-V+2*A,P=Mt(d.Math.ceil(p)),V+=(p-P)/2-A,v=B(o.O7,Ft,6,P,15,1),y=0;y<P;y++)v[y]=t.c[1]+A;for(T=0;T<t.b[1].b.length;T++)for(E=t.b[1].b[T]-V,S=Mt(E-A),u=d.Math.min(Mt(E+A),P-1),k=S;k<=u;k++)v[k]>t.b[1].a[T]&&(v[k]=t.b[1].a[T]);for(c=0;c<P;c++)v[c]-=A;for(i=t.c[0]-t.c[1],b=0;b<t.b[0].a.length;b++)g=Mt(t.b[0].b[b]-V),g>=0&&g<v.length&&i<t.b[0].a[b]-v[g]&&(i=t.b[0].a[b]-v[g]);for(Q=0;Q<t.b[1].a.length;Q++)t.b[1].a[Q]+=i;e&&(X.sB(t.b[0],t.c[0],t.d[0],r[0]-v0),X.sB(t.b[1],t.c[0],t.d[0],r[0]-v0));},X.WA=function(t,A){var e,r;for(this.b=B(o.m9,R1,47,2,0,1),this.b[0]=t,this.b[1]=A,this.c=B(o.O7,Ft,6,2,15,1),this.d=B(o.O7,Ft,6,2,15,1),this.a=B(o.Q7,H,6,2,15,1),e=0;e<2;e++){for(r=0;r<this.b[e].e.length;r++)this.c[e]+=X.jB(this.b[e],r),this.d[e]+=X.kB(this.b[e],r);this.a[e]=this.b[e].e.length;}},X.XA=function(t,A,e,r){this.b=B(o.m9,R1,47,2,0,1),this.b[0]=t,this.b[1]=A,this.c=B(o.O7,Ft,6,2,15,1),this.d=B(o.O7,Ft,6,2,15,1),this.c[0]=X.jB(this.b[0],e),this.d[0]=X.kB(this.b[0],e),this.c[1]=X.jB(this.b[1],r),this.d[1]=X.kB(this.b[1],r),this.a=B(o.Q7,H,6,2,15,1),this.a[0]=1,this.a[1]=1;},x(106,1,{106:1},X.WA,X.XA),o.h9=N(106),X.YA=function(t,A){this.a=t,this.b=A;},X.ZA=function(t,A,e,r){var P,i;this.a=X.$A(t,A,e,r),P=e-t,i=r-A,this.b=d.Math.sqrt(P*P+i*i);},X.$A=function(t,A,e,r){var P,i,D;return i=e-t,D=r-A,D!=0?(P=d.Math.atan(i/D),D<0&&(i<0?P-=A1:P+=A1)):P=i>0?v0:ue,P;},x(41,1,{41:1},X.YA,X.ZA),s.a=0,s.b=0,o.i9=N(41),X._A=function(t){this.a=B(o.Q7,H,6,t,15,1),this.b=B(o.Q7,H,6,t,15,1);},x(247,1,{},X._A),o.j9=N(247),X.aB=function(t,A,e){this.c=t,this.a=A,this.b=e;},x(197,1,{197:1},X.aB),s.a=0,s.b=0,o.k9=N(197),X.cB=function(){X.cB=f1,X.bB=j(F(o.Ocb,1),yt,2,6,["gkvt@@@@LddTTTrbTRTRTRRRRRRRRRRRRRrVRrIh\\IAaQxlY@gRHdJCJcRXlv_CfJx|A\\hRHejiLaQjTje^kSjtFcIhvXmVKMjt{lN{Kavy\\^wGjjjjjjjjjjjjjjjjjjjjjjjjjjjjjjh@@vo@HBC@PhLN@bPhtFKCcpDbILaRhtzCIbsX\\nOO`JDbqDjSKdJeJmQjtz}Ahr[LVkMnpz\\nwGj{PBhBdBlBBBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajijej` !BnkjyVwsVr|iQn|Q|goTZWPIJwbudnRkVYBez]siZymNJZUqNFBqZWxS~iCXVU]SeRjwrtSPAjkvXLpBAZauDPzq]PfMlecrMnkv|@\\SFD`m|mWiEoCXp`SIe_J[l|[XCbloTV`[Gc@FJGopyyoOlFQfUy^w\\Bgz|","gcrt@@@@LdbbbbTRbRbRbRRRRRRRRRRRRVRrVQIA`HtRGAaIxZAHfShTjCIbqylQGKgqdBaXeQJeruBiPitZmFoPZLFSYbvZlVGMnsZ]vWSmr{]UUUUUUUUUUUUUUUUUUUUUUUUUUUUUT@@[G`DAA`HTFG@QHTZCEaqxBQDfPiTZ]AdqYlNWGgpEBQXbUIerEReVhuZ]^`tYMfKUfwX]NW[jkPBhBdBlBbBjBfBnBaBiBeBmBcBkBgBoB`bhbdblbbbjbfbnbabibebmbcbkbgbob`RhRdRlRbRjRfRnRaRiReRmRcRkRgRoR`rhrdrlrbrjrfrnrarirermrcrkrgror`JhJdJlJbJjJfJnJaJiJeJmJcJkJgJoJ`jhjdjljbjjjfjnjajij` !B^cR]`]Fm]QkfljE\\p\x7FUVfgOmFXsQe_gXPyXis_wF|vUUX_XbxpzU]HUFgYViwFo~@uemc@}~T\x7FIEPioYVwr]JnM~[ZEC\\g}~o_pUfdo~irsklTLiyVJshnw^iVAsZ`_~}PYkckURH{FYMImFaRaccUlCZSHMfP","dml@@Dje^VGiyZjjjh@vtHSBinFU@ !BPTCTy[skMzUPF`AJbBixEZHS[Il","dml@@DjYVvGiyZjjjh@vtHSBinFU@ !BwLo~BJ~UquhXBinZ\\ykA@F_eMrT","dml@@LdfbbQX^fUZjjj`C[PaLJfxYT !BzxIHVc{OiJVRpprePho~]}y\x7FwLl","deL@@DjUYkfEijjjj@MeBDpj[ad !B\x7FaA\x7FMVr[AvkKzm_jKvVbD{sk","dil@@LddTQRl[NX^Fjjjj@MiBDpj[a@ !BPfL@\x7Fox@M~T@\x7Fox@\x7F`C~@@","daL@@DjYtKJqjynjjjj@MaBDpj[` !B`bL@_gx@@Gy~@Gx@_`@"]);},X.dB=function(){X.cB();var t,A,e,r,P,i,D;for(U.Mj.call(this),i=new n.yu(),e=X.bB,r=0,P=e.length;r<P;++r)A=e[r],t=n.Pq(new n.Xq(),A),n.ru(i,t),D=new X.wB(t),D.a=n.Ul(D.b),w.iAb(this.a,D);},x(369,20,We,X.dB),o.l9=N(369),X.eB=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z;if(t.e.length==1)return 0;for(O=P+d.Math.sqrt(r),L=B(o.O7,Ft,6,36,15,1),b=0;b<t.e.length;b++)i=X.$A(A,e,t.a[b],t.b[b]),u=X.fB(Jt(b1(d.Math.round(i*36/N1)))),y=A-t.a[b],T=e-t.b[b],Qt=y*y+T*T,L[u]<Qt&&(L[u]=Qt);for(q=-1,E=-1,g=0;g<36;g++)L[g]=d.Math.sqrt(L[g]),q<L[g]&&(q=L[g],E=g);for(tt=E-18<0?E-18+36:E-18>=36?E-18-36:E-18,k=0;k<=18;k++)L[tt+k<0?tt+k+36:tt+k>=36?tt+k-36:tt+k]+=.01*k,L[tt-k<0?tt-k+36:tt-k>=36?tt-k-36:tt-k]+=.01*k;for(z=B(o.O7,Ft,6,9,15,1),Q=B(o.O7,Ft,6,9,15,1),v=1;v<9;v++)z[v]=d.Math.sin(v*r9),Q[v]=d.Math.cos(v*r9);for(G=s0,V=-1,D=0;D<36;D++)if(!(L[D]>=G)){for(p=L[D],c=1;c<9;c++){for(S=-1;S<=1&&(K=D+S*c<0?D+S*c+36:D+S*c>=36?D+S*c-36:D+S*c,!(!(L[K]*Q[c]<=p)&&(l=Q[c]*d.Math.min(L[K],O/z[c]),p<l&&(p=l,G<=p))));S+=2);if(G<=p)break;}G>p&&(G=p,V=D);}return N1*V/36;},X.fB=function(t){return t<0?t+36:t>=36?t-36:t;},X.gB=function(t,A){var e,r,P;if(A.e.length!=t.e.length)return  false;for(r=X.oB(t),P=X.oB(A),e=0;e<r.length;e++)if(r[e]!=P[e])return  false;return  true;},X.hB=function(t,A,e){var r,P,i;for(i=0;i<t.e.length;i++)P=d.Math.sqrt((t.a[i]-A)*(t.a[i]-A)+(t.b[i]-e)*(t.b[i]-e)),r=0-X.$A(A,e,t.a[i],t.b[i]),t.a[i]=A+P*d.Math.sin(r),t.b[i]=e+P*d.Math.cos(r);},X.iB=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V;if(t.d==null&&(t.d=B(o.Q7,E1,7,t.j.r,0,2)),t.d[A]==null){for(c=B(o.Q7,H,6,t.e.length,15,1),p=B(o.vfb,Tt,6,t.j.q,16,1),P=n.Zl(t.j,0,A),i=n.Zl(t.j,1,A),c[0]=P,p[P]=true,L=0,b=0;L<=b;){for(k=0;k<n.Lo(t.j,c[L]);k++)D=n.Zo(t.j,c[L],k),!p[D]&&D!=i&&(c[++b]=D,p[D]=true);if(L==b)break;++L;}if(T=b+1>(t.e.length/2|0),t.i){for(Q=false,u=false,k=0;k<t.e.length;k++)r=t.e[k],n.Km(t.j,r)&&r!=P&&r!=i&&(p[t.e[k]]?Q=true:u=true);Q!=u&&(T=Q);}for(l=2,t.d[A]=B(o.Q7,H,6,T?t.e.length-b:b+2,15,1),v=0;v<t.e.length;v++)t.e[v]==P?t.d[A][T?0:1]=v:t.e[v]==i?t.d[A][T?1:0]=v:T^p[t.e[v]]&&(t.d[A][l++]=v);}for(q=t.a[t.d[A][0]],V=t.b[t.d[A][0]],E=X.$A(q,V,t.a[t.d[A][1]],t.b[t.d[A][1]]),g=2;g<t.d[A].length;g++)S=t.d[A][g],y=d.Math.sqrt((t.a[S]-q)*(t.a[S]-q)+(t.b[S]-V)*(t.b[S]-V)),e=2*E-X.$A(q,V,t.a[S],t.b[S]),t.a[S]=q+y*d.Math.sin(e),t.b[S]=V+y*d.Math.cos(e);},X.jB=function(t,A){return t.a[A];},X.kB=function(t,A){return t.b[A];},X.lB=function(t){var A,e,r,P,i,D,u,Q;for(t.c=0,e=new U.Mj(),P=1;P<t.e.length;P++)for(i=0;i<P;i++)u=d.Math.abs(t.a[P]-t.a[i]),Q=d.Math.abs(t.b[P]-t.b[i]),r=d.Math.sqrt(u*u+Q*Q),r<.8&&(A=B(o.Q7,H,6,2,15,1),A[0]=t.e[P],A[1]=t.e[i],w.iAb(e.a,A)),D=1-d.Math.min(r,1),t.c+=D*D;return e;},X.mB=function(t,A){return t.e[A];},X.nB=function(t,A){var e;for(e=0;e<t.e.length;e++)if(A==t.e[e])return e;return  -1;},X.oB=function(t){return t.n==null&&(t.n=U.zrb(t.e,t.e.length),w.lAb(t.n,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[])))),t.n;},X.pB=function(t,A){var e;for(e=0;e<t.e.length;e++)if(A==t.e[e])return  true;return  false;},X.qB=function(t){var A,e,r,P,i,D;for(r=0,i=0;i<t.e.length;i++)for(A=t.e[i],e=n.Lo(t.j,A),D=0;D<e;D++)n.Zo(t.j,A,D)>A&&++r;for(t.f=B(o.Q7,H,6,r,15,1),t.g=B(o.Q7,H,6,t.j.q,15,1),r=0,P=0;P<t.e.length;P++)for(A=t.e[P],e=n.Lo(t.j,A),t.g[A]=P,D=0;D<e;D++)n.Zo(t.j,A,D)>A&&(t.f[r++]=n._o(t.j,A,D));},X.rB=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;for(b=t.a[A],S=t.b[A],i=B(o.i9,R1,41,4,0,1),L=0,y=0;y<t.f.length&&!(L>=4);y++)if(!(A==t.g[n.Zl(t.j,0,t.f[y])]||A==t.g[n.Zl(t.j,1,t.f[y])])){if(g=t.a[t.g[n.Zl(t.j,0,t.f[y])]],p=t.b[t.g[n.Zl(t.j,0,t.f[y])]],k=t.a[t.g[n.Zl(t.j,1,t.f[y])]],E=t.b[t.g[n.Zl(t.j,1,t.f[y])]],u=d.Math.sqrt((g-b)*(g-b)+(p-S)*(p-S)),Q=d.Math.sqrt((k-b)*(k-b)+(E-S)*(E-S)),P=d.Math.sqrt((k-g)*(k-g)+(E-p)*(E-p)),u<P&&Q<P){g==k?(D=d.Math.abs(b-g),D<.5&&(i[L++]=new X.YA(X.$A(g,S,b,S),(.5-D)/2))):p==E?(D=d.Math.abs(S-p),D<.5&&(i[L++]=new X.YA(X.$A(b,p,b,S),(.5-D)/2))):(T=(E-p)/(k-g),c=-1/T,e=p-T*g,r=S-c*b,v=(r-e)/(T-c),q=T*v+e,D=d.Math.sqrt((v-b)*(v-b)+(q-S)*(q-S)),D<.5&&(i[L++]=new X.YA(X.$A(v,q,b,S),(.5-D)/2)));continue;}if(u<.5){i[L++]=new X.YA(X.$A(g,p,b,S),(.5-u)/2);continue;}if(Q<.5){i[L++]=new X.YA(X.$A(k,E,b,S),(.5-Q)/2);continue;}}L>0&&(l=X.DA(i,L),t.a[A]+=l.b*d.Math.sin(l.a),t.b[A]+=l.b*d.Math.cos(l.a));},X.sB=function(t,A,e,r){var P,i,D;for(D=0;D<t.e.length;D++)i=d.Math.sqrt((t.a[D]-A)*(t.a[D]-A)+(t.b[D]-e)*(t.b[D]-e)),P=X.$A(A,e,t.a[D],t.b[D])+r,t.a[D]=A+i*d.Math.sin(P),t.b[D]=e+i*d.Math.cos(P);},X.tB=function(t,A,e){var r;for(r=0;r<t.e.length;r++)t.a[r]+=A,t.b[r]+=e;},X.uB=function(t,A,e){this.j=t,this.i=e,this.e=B(o.Q7,H,6,A,15,1),this.k=B(o.Q7,H,6,A,15,1),this.a=B(o.O7,Ft,6,A,15,1),this.b=B(o.O7,Ft,6,A,15,1);},X.vB=function(t){var A,e;for(this.j=t.j,this.i=t.i,this.e=B(o.Q7,H,6,t.e.length,15,1),this.k=B(o.Q7,H,6,t.e.length,15,1),this.a=B(o.O7,Ft,6,t.e.length,15,1),this.b=B(o.O7,Ft,6,t.e.length,15,1),e=0;e<t.e.length;e++)this.e[e]=t.e[e],this.k[e]=t.k[e],this.a[e]=t.a[e],this.b[e]=t.b[e];if(t.f!=null)for(this.f=B(o.Q7,H,6,t.f.length,15,1),A=0;A<t.f.length;A++)this.f[A]=t.f[A];if(t.g!=null)for(this.g=B(o.Q7,H,6,t.g.length,15,1),A=0;A<t.g.length;A++)this.g[A]=t.g[A];},x(47,1,{47:1},X.uB,X.vB),s.c=0,s.i=false,o.m9=N(47),X.wB=function(t){this.b=t,this.c=false;},x(246,1,{246:1},X.wB),s.a=0,s.c=false,o.n9=N(246),gA.xB=function(t){t.c=new U.Mj();},gA.yB=function(t,A,e,r,P,i){var D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt;for(p=-1,G=0,V=0,O=0,k=0,v=0,S=0,tt=0,Qt=0,G=0,L=0;L<t.d;L++)G+=r[L]*r[L];if(G=d.Math.sqrt(G),G>i)for(y=0;y<t.d;y++)r[y]*=i/G;for(V=0,T=0;T<t.d;T++)V+=r[T]*t.e[T];if(V>=0)return p;for(O=0,c=0;c<t.d;c++)K=d.Math.abs(r[c])/d.Math.max(d.Math.abs(A[c]),1),K>O&&(O=K);for(S=1e-7/O,k=1,g=0;g<1e3;){if(k<S){p=1;break;}for(b=0;b<t.d;b++)P[b]=A[b]+k*r[b];if(t.k=Z.xC(t,P),t.k-e<=we*k*V)return 0;g==0?tt=-V/(2*(t.k-e-V)):(E=t.k-e-k*V,q=Qt-e-v*V,D=(E/(k*k)-q/(v*v))/(k-v),u=(-v*E/(k*k)+k*q/(v*v))/(k-v),D==0?tt=-V/(2*u):(Q=u*u-3*D*V,Q<0?tt=.5*k:u<=0?tt=(-u+d.Math.sqrt(Q))/(3*D):tt=-V/(u+d.Math.sqrt(Q))),tt>.5*k&&(tt=.5*k)),v=k,Qt=t.k,k=d.Math.max(tt,.1*k),++g;}for(l=0;l<t.d;l++)P[l]=A[l];return p;},gA.zB=function(t,A,e){var r,P,i,D;for(D=1,P=0;P<t.g.q;P++)t.i[3*P]=n.Ql(t.g,P),t.i[3*P+1]=n.Rl(t.g,P),t.i[3*P+2]=n.Sl(t.g,P);if(D=gA.AB(t,A,e),D==0)for(r=0;r<t.g.q;r++)n.zn(t.g,r,t.i[3*r]),n.An(t.g,r,t.i[3*r+1]),n.Bn(t.g,r,t.i[3*r+2]);for(i=new U.rrb(t.c);i.a<i.c.a.length;)U.qrb(i),null.zd();return D;},gA.AB=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt;for(t.e=B(o.O7,Ft,6,t.d,15,1),r=B(o.O7,Ft,6,t.d,15,1),l=B(o.O7,Ft,6,t.d,15,1),K=B(o.O7,Ft,6,t.d,15,1),wt=B(o.O7,Ft,6,t.d,15,1),S=B(o.O7,Ft,6,t.d*t.d,15,1),y=0;y<t.d;y++)K[y]=t.i[y];for(u=Z.xC(t,t.i),Z.yC(t),tt=0,T=0;T<t.d;T++)S[T*t.d+T]=1,wt[T]=-t.e[T],tt+=t.i[T]*t.i[T];for(G=100*d.Math.max(d.Math.sqrt(tt),t.d),it=(h.Pkb(),b1(Date.now())),E=1;E<=A&&!t.f;E++){if(O=gA.yB(t,t.i,u,wt,K,G),O<0)return 2;for(u=t.k,ot=0,c=0;c<t.d;c++)wt[c]=K[c]-t.i[c],t.i[c]=K[c],ut=d.Math.abs(wt[c])/d.Math.max(d.Math.abs(t.i[c]),1),ut>ot&&(ot=ut),r[c]=t.e[c];if(ot<12e-8)return 0;for(Q=Z.yC(t),ot=0,Dt=d.Math.max(t.k*Q,1),b=0;b<t.d;b++)Ut=d.Math.abs(t.e[b])*d.Math.max(d.Math.abs(t.i[b]),1),ot=d.Math.max(ot,Ut),r[b]=t.e[b]-r[b];if(ot/=Dt,ot<e)return 0;for(P=0,D=0,Qt=0,z=0,g=0;g<t.d;g++){for(p=g*t.d,l[g]=0,q=0;q<t.d;q++)l[g]+=S[p+q]*r[q];P+=r[g]*wt[g],D+=r[g]*l[g],Qt+=r[g]*r[g],z+=wt[g]*wt[g];}if(P>d.Math.sqrt(3e-8*Qt*z)){for(P=1/P,i=1/D,k=0;k<t.d;k++)r[k]=P*wt[k]-i*l[k];for(v=0;v<t.d;v++)for(p=v*t.d,q=v;q<t.d;q++)S[p+q]+=P*wt[v]*wt[q]-i*l[v]*l[q]+D*r[v]*r[q],S[q*t.d+v]=S[p+q];}for(L=0;L<t.d;L++)for(p=L*t.d,wt[L]=0,q=0;q<t.d;q++)wt[L]-=S[p+q]*t.e[q];if(lt=b1(Date.now()),bt=x1(lt,it),Bn(bt,t.j)){for(V=new U.rrb(t.c);V.a<V.c.a.length;)U.qrb(V),null.zd();it=lt;}}return 1;},gA.BB=function(t){var A,e,r;for(gA.xB(this),r=0,A=0;A<t.f;A++)r+=n.op(t,A);if(r>0)throw st(new h._ib("molecule needs explicit hydrogen atoms for force field calculations"));for(this.g=t,this.d=3*t.q,this.e=B(o.O7,Ft,6,this.d,15,1),this.i=B(o.O7,Ft,6,this.d,15,1),this.f=false,this.j=20,e=0;e<t.q;e++)this.i[3*e]=t.J[e].a,this.i[3*e+1]=t.J[e].b,this.i[3*e+2]=t.J[e].c;},x(275,1,{}),s.d=0,s.f=false,s.j=0,s.k=0,o.o9=N(275),Z.CB=function(t,A,e,r,P){this.f=A,this.a=e,this.b=r,this.c=P,this.d=rt.bE(t.b,A.a[r]),this.g=rt.WD(t.a,A,e,r,P),this.e=rt.VD(t.a,A,e,r,P);},Z.DB=function(t,A){var e,r,P,i,D,u;for(e=new U.Mj(),r=0;r<A.q;r++)if(A.e[r]>1)for(P=0;P<A.e[r];P++)for(D=A.i[r][P],i=P+1;i<A.e[r];i++)u=A.i[r][i],U.sj(e,new Z.CB(t,A,D,r,u));return e;},x(360,1,Un,Z.CB),s.cc=function(t,A,e,r){var P,i,D;return r&&(n.Tl(this.f,this.a)==1||n.Tl(this.f,this.c)==1)?0:(D=Z.DD(new Z.PD(t,this.b,this.a),new Z.PD(t,this.b,this.c)),P=D*yn-this.g,i=this.d?143.9325*this.e*(1+d.Math.cos(D)):P9*this.e*P*P*(1+-6981317e-9*P),i);},s.dc=function(t,A){var e,r,P,i,D,u,Q,l,L,y;Q=Z.ID(new Z.PD(t,this.b,this.a)),l=Z.ID(new Z.PD(t,this.b,this.c)),D=Z.HD(new Z.PD(t,this.b,this.a)),u=Z.HD(new Z.PD(t,this.b,this.c)),r=(Q.a*l.a+Q.b*l.b+Q.c*l.c)/(d.Math.sqrt(Q.a*Q.a+Q.b*Q.b+Q.c*Q.c)*d.Math.sqrt(l.a*l.a+l.b*l.b+l.c*l.c)),y=1-r*r,L=ge,y>0&&(L=d.Math.sqrt(y)),e=yn*d.Math.acos(r)-this.g,i=DA*this.e*e*(1+-0.0104719755*e),this.d&&(i=-143.9325*this.e*L),P=j(F(o.O7,1),Ft,6,15,[1/D*(l.a-r*Q.a),1/D*(l.b-r*Q.b),1/D*(l.c-r*Q.c),1/u*(Q.a-r*l.a),1/u*(Q.b-r*l.b),1/u*(Q.c-r*l.c)]),A[3*this.a]+=i*P[0]/-L,A[3*this.a+1]+=i*P[1]/-L,A[3*this.a+2]+=i*P[2]/-L,A[3*this.b]+=i*(-P[0]-P[3])/-L,A[3*this.b+1]+=i*(-P[1]-P[4])/-L,A[3*this.b+2]+=i*(-P[2]-P[5])/-L,A[3*this.c]+=i*P[3]/-L,A[3*this.c+1]+=i*P[4]/-L,A[3*this.c+2]+=i*P[5]/-L;},s.a=0,s.b=0,s.c=0,s.d=false,s.e=0,s.g=0,o.p9=N(360),Z.gC=function(t){h.fC.call(this,t);},x(185,33,{185:1,4:1,19:1,33:1,27:1},Z.gC),o.q9=N(185),Z.hC=function(){h.dC.call(this);},x(322,33,m0,Z.hC),o.r9=N(322),Z.iC=function(t,A,e){Z.jC.call(this,t,A,A.D[0][e],A.D[1][e]);},Z.jC=function(t,A,e,r){this.d=A,this.a=e,this.b=r,this.e=rt.pE(t.d,A,e,r),this.c=rt.oE(t.d,A,e,r);},Z.kC=function(t,A){var e,r;for(r=new U.Mj(),e=0;e<A.r;e++)U.sj(r,new Z.iC(t,A,e));return r;},x(237,1,Un,Z.iC),s.cc=function(t,A,e,r){var P,i,D;return r&&(n.Tl(this.d,this.a)==1||n.Tl(this.d,this.b)==1)?0:(i=Z.FD(new Z.OD(t,this.a),new Z.OD(t,this.b)),P=(i-this.e)*(i-this.e),D=71.96625*this.c*P*(1+-2*(i-this.e)+2.3333333333333335*P),D);},s.dc=function(t,A){var e,r,P,i;if(r=Z.FD(new Z.OD(t,this.a),new Z.OD(t,this.b)),P=r-this.e,e=143.9325*this.c*P*(1+-3*P+4.666666666666667*P*P),r>0)for(i=0;i<3;i++)A[3*this.a+i]+=e*(t[3*this.a+i]-t[3*this.b+i])/r,A[3*this.b+i]-=e*(t[3*this.a+i]-t[3*this.b+i])/r;},s.a=0,s.b=0,s.c=0,s.e=0,o.s9=N(237),Z.nC=function(t){var A,e,r,P,i,D,u,Q,l;A=null;try{for(A=new Rt.ehb(new Rt.jhb((P=_t.cBb(t),q1.Cnb(),P))),Q=h.Mib(h.skb(Rt.dhb(A)),r1,qt),r=h.okb(h.skb(Rt.dhb(A)),","),l=U1(o.Icb,[b0,R1],[8,1],5,[Q,r.length],2),D=0;(i=Rt.dhb(A))!=null&&D<Q;D++)for(u=h.okb(h.skb(i),","),e=0;e<r.length;e++)switch(h.Ujb(r[e],0)){case 105:a1(l[D],e,h.qjb(h.Mib(h.skb(u[e]),r1,qt)));break;case 102:a1(l[D],e,w.SAb(h.Lib(h.skb(u[e]))));break;case 99:a1(l[D],e,r2(h.Ujb(h.mkb(h.mkb(h.skb(u[e]),"'"),'"'),0)));}return l;}catch(L){if(L=C1(L),Zt(L,75))h.Pkb();else throw st(L);}return U1(o.Icb,[b0,R1],[8,1],5,[1,1],2);},Z.oC=function(t){var A,e,r,P,i,D;for(D=Z.nC(t),i=D.length,A=D[0].length,P=U1(o.Q7,[E1,H],[7,6],15,[i,A],2),e=0;e<i;e++)for(r=0;r<A;r++)P[e][r]=D[e][r].a;return P;},Z.pC=function(t,A,e,r,P,i,D,u){this.e=t,this.a=A,this.b=e,this.f=r,this.d=D,this.c=P*i/u;},Z.qC=function(t,A,e,r,P,i){var D,u,Q,l,L;for(u=new U.Mj(),D=P1.CF(t,A),Q=0;Q<A.q;Q++)for(l=0;l<Q+1;l++)L=Z.ZC(e,new Z.iD(Q,l)),(L==(Z.eD(),Z._C)||L==Z.dD)&&d.Math.abs(D[Q])>Be&&d.Math.abs(D[l])>Be&&Z.HD(new Z.MD(A,Q,l))<r&&U.sj(u,new Z.pC(A,Q,l,L,D[Q],D[l],P,i));return u;},x(361,1,Un,Z.pC),s.cc=function(t,A,e,r){var P,i,D;return r&&(n.Tl(this.e,this.a)==1||n.Tl(this.e,this.b)==1)?0:(i=Z.HD(new Z.PD(t,this.a,this.b)),P=i+.05,this.d&&(P*=P),D=332.0716*this.c/P*(this.f==(Z.eD(),Z._C)?.75:1),D);},s.dc=function(t,A){var e,r,P,i,D;for(i=Z.HD(new Z.PD(t,this.a,this.b)),e=i+.05,e*=this.d?e*e:e,r=-332.0716*(this.d?2:1)*this.c/e*(this.f==(Z.eD(),Z._C)?.75:1),D=0;D<3;D++)P=.02,i>0&&(P=r*(t[3*this.a+D]-t[3*this.b+D])/i),A[3*this.a+D]+=P,A[3*this.b+D]-=P;},s.a=0,s.b=0,s.c=0,s.d=false,o.t9=N(361),Z.uC=function(){Z.uC=f1,Z.tC=new U.pub();},Z.vC=function(t){t.a=new U.Mj();},Z.wC=function(t){return Z.xC(t,t.i);},Z.xC=function(t,A){var e,r,P,i,D;for(D=0,r=1,i=new U.rrb(t.a);i.a<i.c.a.length;)P=U.qrb(i),e=P.cc(A,null,""+r,false),e!=0&&(D+=e,++r);return D;},Z.yC=function(t){var A,e,r,P,i,D;for(t.e=B(o.O7,Ft,6,t.d,15,1),e=new U.rrb(t.a);e.a<e.c.a.length;)A=U.qrb(e),A.dc(t.i,t.e);for(D=-1e8,r=.1,i=0;i<t.d;i++)t.e[i]*=r,t.e[i]>D&&(D=t.e[i]);if(D>10){for(;D*r>10;)r*=.5;for(P=0;P<t.d;P++)t.e[P]*=r;}return r;},Z.zC=function(t){Z.uC(),Z.AC.call(this,t,yr,new U.pub());},Z.AC=function(t,A,e){Z.uC();var r,P,i,D,u;gA.BB.call(this,t),Z.vC(this),this.b=new Z.CC(t),n.bw(this.g,7),u=U.oob(Z.tC,A),i=U.Yub(e.c,i9)?h.Rib(U.Zub(e.c,i9)):100,r=U.Yub(e.c,f9)?h.Rib(U.Zub(e.c,f9)):1,P=U.Yub(e.c,D9)&&h.Oc(U.Zub(e.c,D9),"distance"),D=new Z.$C(this.b),(!U.Yub(e.c,u9)||h.Hhb(U.Zub(e.c,u9)))&&U.uj(this.a,Z.DB(u,this.b)),(!U.Yub(e.c,Q9)||h.Hhb(U.Zub(e.c,Q9)))&&U.uj(this.a,Z.kC(u,this.b)),(!U.Yub(e.c,o9)||h.Hhb(U.Zub(e.c,o9)))&&U.uj(this.a,Z.qC(u,this.b,D,i,P,r)),(!U.Yub(e.c,l9)||h.Hhb(U.Zub(e.c,l9)))&&U.uj(this.a,Z.EC(u,this.b)),(!U.Yub(e.c,L9)||h.Hhb(U.Zub(e.c,L9)))&&U.uj(this.a,Z.oD(u,this.b)),(!U.Yub(e.c,s9)||h.Hhb(U.Zub(e.c,s9)))&&U.uj(this.a,Z.uD(u,this.b)),(!U.Yub(e.c,U9)||h.Hhb(U.Zub(e.c,U9)))&&U.uj(this.a,Z.AD(u,this.b,D,i));},Z.BC=function(t,A){Z.uC(),U.pob(Z.tC,t)||U.rob(Z.tC,t,A);},x(178,275,{},Z.zC,Z.AC),o.u9=N(178),Z.CC=function(t){n.Rk();var A,e,r,P,i;for(n.vw.call(this,t),n.kp(this,n.Ko(this)),i=(n.bw(this,7),this.p),this.b=B(o.x9,H,6,i.j.a.length,15,1),U.Frb(this.b),A=false,e=true;!A&&e;)for(A=true,e=false,P=0;P<i.j.a.length;P++)this.b[P]==2&&(this.b[P]=P1.zF(this,P),this.b[P]!=2&&(e=true)),this.b[P]==2&&(A=false);if(!A)throw st(new Z.hC());for(this.a=B(o.Q7,H,6,this.q,15,1),r=0;r<this.a.length;r++)if(this.a[r]=-1,this.a[r]=P1.tF(this,r),this.a[r]==0)throw st(new Z.gC("Couldn't assign an atom type to atom "+r+" ("+n.Lk[this.C[r]]+")"));},x(276,26,fi,Z.CC),o.v9=N(276),Z.DC=function(t,A,e,r,P,i){this.f=A,this.d=e,this.a=r,this.b=P,this.c=i,this.e=rt.QE(t.k,A,e,r,P,i);},Z.EC=function(t,A){var e,r,P,i,D;for(D=new U.Mj(),i=0;i<A.q;i++)A.e[i]==3&&(e=A.i[i][0],r=A.i[i][1],P=A.i[i][2],U.sj(D,new Z.DC(t,A,i,e,r,P)),U.sj(D,new Z.DC(t,A,i,e,P,r)),U.sj(D,new Z.DC(t,A,i,r,P,e)));return D;},x(192,1,Un,Z.DC),s.cc=function(t,A,e,r){var P,i,D,u,Q,l;return r&&(n.Tl(this.f,this.a)==1||n.Tl(this.f,this.b)==1||n.Tl(this.f,this.c)==1)?0:(u=Z.ID(new Z.PD(t,this.d,this.a)),Q=Z.ID(new Z.PD(t,this.d,this.b)),l=Z.ID(new Z.PD(t,this.d,this.c)),D=Z.ID(new Z.KD(u.b*Q.c-u.c*Q.b,u.c*Q.a-u.a*Q.c,u.a*Q.b-u.b*Q.a)),P=yn*d.Math.asin(D.a*l.a+D.b*l.b+D.c*l.c),i=P9*this.e*P*P,i);},s.dc=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O;for(T=new Z.PD(t,this.d,this.a),c=new Z.PD(t,this.d,this.b),b=new Z.PD(t,this.d,this.c),u=d.Math.sqrt(T.a*T.a+T.b*T.b+T.c*T.c),Q=d.Math.sqrt(c.a*c.a+c.b*c.b+c.c*c.c),l=d.Math.sqrt(b.a*b.a+b.b*b.b+b.c*b.c),T=Z.ID(T),c=Z.ID(c),b=Z.ID(b),y=Z.ID(Z.ED(new Z.KD(-T.a,-T.b,-T.c),c)),g=b.a*y.a+b.b*y.b+b.c*y.c,P=1-g*g,r=d.Math.max(P>0?d.Math.sqrt(P):0,ge),e=yn*d.Math.asin(g),i=T.a*c.a+T.b*c.b+T.c*c.c,v=d.Math.max(1-i*i,ge),k=d.Math.max(v>0?d.Math.sqrt(v):0,ge),D=DA*this.e*e,S=new Z.KD(b.b*c.c-b.c*c.b,b.c*c.a-b.a*c.c,b.a*c.b-b.b*c.a),p=new Z.KD(T.b*b.c-T.c*b.b,T.c*b.a-T.a*b.c,T.a*b.b-T.b*b.a),E=new Z.KD(c.b*T.c-c.c*T.b,c.c*T.a-c.a*T.c,c.a*T.b-c.b*T.a),q=r*k,V=g/(r*v),G=j(F(o.O7,1),Ft,6,15,[(S.a/q-(T.a-c.a*i)*V)/u,(S.b/q-(T.b-c.b*i)*V)/u,(S.c/q-(T.c-c.c*i)*V)/u]),K=j(F(o.O7,1),Ft,6,15,[(p.a/q-(c.a-T.a*i)*V)/Q,(p.b/q-(c.b-T.b*i)*V)/Q,(p.c/q-(c.c-T.c*i)*V)/Q]),O=j(F(o.O7,1),Ft,6,15,[(E.a/q-b.a*g/r)/l,(E.b/q-b.b*g/r)/l,(E.c/q-b.c*g/r)/l]),L=0;L<3;L++)A[3*this.a+L]+=D*G[L],A[3*this.d+L]+=-D*(G[L]+K[L]+O[L]),A[3*this.b+L]+=D*K[L],A[3*this.c+L]+=D*O[L];},s.a=0,s.b=0,s.c=0,s.d=0,s.e=0,o.w9=N(192),Z.HC=function(t){return t<3?0:t<11?1:t<19?2:t<37?3:t<55?4:0;},Z.IC=function(t){var A;return A=0,t==2?A=1:t>=3&&t<=10?A=2:t>=11&&t<=18?A=3:t>=19&&t<=36?A=4:t>=37&&t<=54&&(A=5),(t>=21&&t<=30||t>=39&&t<=48)&&(A*=10),A;},h.JC=function(t,A){return t.b-A.b;},h.KC=function(t){return t.a!=null?t.a:""+t.b;},h.LC=function(t){return t.b;};function je(f,t){this.a=f,this.b=t;}x(56,1,{4:1,32:1,56:1}),s.Sb=function(t){return h.JC(this,t);},s.compareTo=function(t){return h.JC(this,t);},s.equals=function(t){return this===t;},s.mb=function(f){return this.equals(f);},s.hashCode=function(){return w.yAb(this);},s.ob=function(){return this.hashCode();},s.name=function(){return h.KC(this);},s.ordinal=function(){return h.LC(this);},s.toString=function(){return this.a!=null?this.a:""+this.b;},s.pb=function(){return this.toString();},s.b=0,o.wcb=N(56),Z.UC=function(){return j(F(o.x9,1),H,6,15,[0,1,2]);},o.x9=zA(null,Z.UC),Z.VC=function(t,A,e,r,P,i){var D,u,Q,l;if(e=d.Math.max(0,e),r=d.Math.min(i.lc(),r),D=(r-e)/2|0,Q=e,u=r,i.kc(Q,t)>A||i.kc(u-1,t)<A)return  -1;for(;r>=e;){if(D=e+((r-e)/2|0),l=i.kc(D,t),l==A){if(P&&D>Q&&i.kc(D-1,t)==A)r=D;else if(!P&&D<u-1&&i.kc(D+1,t)==A)e=D;else return D;}else l>A?r=D:l<A&&(e=D);if(r-e==1&&i.kc(e,t)<A&&i.kc(r,t)>A)break;}return  -1;},Z.WC=function(t,A){return Z.VC(0,t,0,A.a.length,true,A);},Z.XC=function(t,A,e){var r,P,i;if(t.length!=A.length)return  -1;for(i=0,r=e.lc(),P=0;P<t.length-1;P++)if(i=Z.VC(t[P],A[P],i,r+1,true,e),r=Z.VC(t[P],A[P],i,r+1,false,e),i==-1||r==-1)return  -1;return Z.VC(t[P],A[P],i,r+1,true,e);},Z.YC=function(t){t.a=new U.pub();},Z.ZC=function(t,A){return U.nob(t.a,A)!=null?U.nob(t.a,A):(Z.eD(),Z.dD);},Z.$C=function(t){var A,e,r,P,i,D,u,Q,l,L;for(Z.YC(this),A=0;A<t.q;A++)for(U.qob(this.a,new Z.iD(A,A),(Z.eD(),Z.aD)),e=0;e<t.e[A];e++)for(Q=t.i[A][e],u=new Z.iD(A,Q),U.qob(this.a,u,Z.cD),r=0;r<t.e[Q];r++)for(l=t.i[Q][r],D=new Z.iD(A,l),(!U.job(this.a,D)||F1(U.nob(this.a,D))===F1(Z._C))&&U.qob(this.a,D,Z.bD),P=0;P<t.e[l];P++)L=t.i[l][P],i=new Z.iD(A,L),U.job(this.a,i)||U.qob(this.a,i,Z._C);},x(277,1,{},Z.$C),o.z9=N(277),Z.eD=function(){Z.eD=f1,Z.aD=new Z.fD("ONE_ONE",0),Z.cD=new Z.fD("ONE_TWO",1),Z.bD=new Z.fD("ONE_THREE",2),Z._C=new Z.fD("ONE_FOUR",3),Z.dD=new Z.fD("ONE_X",4);},Z.fD=function(t,A){je.call(this,t,A);},Z.gD=function(){return Z.eD(),j(F(o.y9,1),y0,108,0,[Z.aD,Z.cD,Z.bD,Z._C,Z.dD]);},x(108,56,{108:1,4:1,32:1,56:1},Z.fD),o.y9=zA(108,Z.gD),Z.hD=function(t,A){return t.a>A.a?1:t.a<A.a?-1:t.b>A.b?1:t.b<A.b?-1:0;},Z.iD=function(t,A){this.a=t>A?t:A,this.b=t>A?A:t;},x(99,1,{99:1,32:1},Z.iD),s.Sb=function(t){return Z.hD(this,t);},s.mb=function(t){var A;return t===this?true:Zt(t,99)?(A=t,this.a==A.a&&this.b==A.b):false;},s.ob=function(){return this.a^this.b;},s.pb=function(){return this.a+","+this.b;},s.a=0,s.b=0,o.A9=N(99),Z.nD=function(t,A,e,r,P){this.f=A,this.a=e,this.b=r,this.c=P,this.j=rt.WD(t.a,A,e,r,P),this.g=rt.pE(t.d,A,e,r),this.i=rt.pE(t.d,A,P,r),this.d=rt.VE(t.n,A,e,r,P),this.e=rt.VE(t.n,A,P,r,e);},Z.oD=function(t,A){var e,r,P,i,D,u,Q,l;for(l=new U.Mj(),e=0;e<A.q;e++)if(r=A.a[e],!(A.e[e]<=1&&rt.bE(t.b,r)))for(i=0;i<A.e[e];i++)for(u=A.i[e][i],D=0;D<A.e[e];D++)Q=A.i[e][D],!(u>Q)&&Z.rD(t,A,u,e,Q)&&(P=new Z.nD(t,A,u,e,Q),(d.Math.abs(P.d)>we||d.Math.abs(P.e)>we)&&w.iAb(l.a,P));return l;},Z.rD=function(t,A,e,r,P){var i;return i=A.a[r],!(rt.bE(t.b,i)||e==P||n.Xo(A,e,r)==-1||n.Xo(A,r,P)==-1);},x(362,1,Un,Z.nD),s.cc=function(t,A,e,r){var P,i,D,u,Q;return r&&(n.Tl(this.f,this.a)==1||n.Tl(this.f,this.c)==1)?0:(P=Z.HD(new Z.PD(t,this.b,this.a)),i=Z.HD(new Z.PD(t,this.b,this.c)),Q=Z.DD(new Z.PD(t,this.b,this.a),new Z.PD(t,this.b,this.c)),u=DA*(Q*yn-this.j),D=u*(P-this.g)*this.d+u*(i-this.i)*this.e,D);},s.dc=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k;L=Z.HD(new Z.PD(t,this.b,this.a)),y=Z.HD(new Z.PD(t,this.b,this.c)),c=Z.ID(new Z.PD(t,this.b,this.a)),b=Z.ID(new Z.PD(t,this.b,this.c)),r=c.a*b.a+c.b*b.b+c.c*b.c,k=1-r*r,g=d.Math.max(k>0?d.Math.sqrt(k):0,ge),e=yn*d.Math.acos(r)-this.j,T=yn*(this.d*(L-this.g)+this.e*(y-this.i)),P=1/L*(b.a-r*c.a),i=1/L*(b.b-r*c.b),D=1/L*(b.c-r*c.c),u=1/y*(c.a-r*b.a),Q=1/y*(c.b-r*b.b),l=1/y*(c.c-r*b.c),A[3*this.a]+=DA*(c.a*this.d*e+P/-g*T),A[3*this.a+1]+=DA*(c.b*this.d*e+i/-g*T),A[3*this.a+2]+=DA*(c.c*this.d*e+D/-g*T),A[3*this.b]+=DA*((-c.a*this.d-b.a*this.e)*e+(-P-u)/-g*T),A[3*this.b+1]+=DA*((-c.b*this.d-b.b*this.e)*e+(-i-Q)/-g*T),A[3*this.b+2]+=DA*((-c.c*this.d-b.c*this.e)*e+(-D-l)/-g*T),A[3*this.c]+=DA*(b.a*this.e*e+u/-g*T),A[3*this.c+1]+=DA*(b.b*this.e*e+Q/-g*T),A[3*this.c+2]+=DA*(b.c*this.e*e+l/-g*T);},s.a=0,s.b=0,s.c=0,s.d=0,s.e=0,s.g=0,s.i=0,s.j=0,o.B9=N(362),Z.sD=function(t,A){this.a=new rt.XD(this),this.e=new rt.wE(),this.b=new rt.gE(),this.c=new rt.lE(),this.d=new rt.qE(this),this.f=new rt.DE(),this.i=new rt.JE(),this.g=new rt.GE(),this.j=new rt.NE(),this.k=new rt.RE(this,t),this.n=new rt.WE(this),this.o=new rt.$E(this,A),this.p=new rt.lF();},x(155,1,{155:1},Z.sD),o.C9=N(155),Z.tD=function(t,A,e,r,P,i){var D;this.a=e,this.b=r,this.c=P,this.d=i,D=rt.ZE(t.o,A,e,r,P,i),this.e=D.a,this.f=D.b,this.g=D.c;},Z.uD=function(t,A){var e,r,P,i,D,u,Q,l,L;for(L=new U.Mj(),e=0;e<A.q;e++)for(D=0;D<A.e[e];D++)for(r=A.i[e][D],u=0;u<A.e[r];u++)if(P=A.i[r][u],e!=P)for(Q=0;Q<A.e[P];Q++)i=A.i[P][Q],!(r==i||e==i)&&i>e&&(l=new Z.tD(t,A,e,r,P,i),(d.Math.abs(l.e)>.001||d.Math.abs(l.f)>.001||d.Math.abs(l.g)>.001)&&w.iAb(L.a,l));return L;},x(363,1,Un,Z.tD),s.cc=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c;return Q=new Z.PD(t,this.a,this.b),l=new Z.PD(t,this.c,this.b),L=new Z.PD(t,this.b,this.c),y=new Z.PD(t,this.d,this.c),T=new Z.KD(Q.b*l.c-Q.c*l.b,Q.c*l.a-Q.a*l.c,Q.a*l.b-Q.b*l.a),c=new Z.KD(L.b*y.c-L.c*y.b,L.c*y.a-L.a*y.c,L.a*y.b-L.b*y.a),D=(T.a*c.a+T.b*c.b+T.c*c.c)/(d.Math.sqrt(T.a*T.a+T.b*T.b+T.c*T.c)*d.Math.sqrt(c.a*c.a+c.b*c.b+c.c*c.c)),P=2*D*D-1,i=D*(2*P-1),u=.5*(this.e*(1+D)+this.f*(1-P)+this.g*(1+i)),u;},s.dc=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;D=j(F(o.F9,1),R1,22,0,[new Z.PD(t,this.b,this.a),new Z.PD(t,this.b,this.c),new Z.PD(t,this.c,this.b),new Z.PD(t,this.c,this.d)]),T=j(F(o.F9,1),R1,22,0,[Z.ED(D[0],D[1]),Z.ED(D[2],D[3])]),r=j(F(o.O7,1),Ft,6,15,[Z.HD(T[0]),Z.HD(T[1])]),!(d.Math.abs(r[0])<Be||d.Math.abs(r[1])<Be)&&(T[0]=Z.ID(T[0]),T[1]=Z.ID(T[1]),e=Z.GD(T[0],T[1]),L=1-e*e,l=L>0?d.Math.sqrt(L):0,u=2*l*e,Q=3*l-4*l*L,i=.5*(-this.e*l+2*this.f*u-3*this.g*Q),y=-i*(d.Math.abs(l)<Be?1/e:1/l),P=j(F(o.O7,1),Ft,6,15,[1/r[0]*(T[1].a-e*T[0].a),1/r[0]*(T[1].b-e*T[0].b),1/r[0]*(T[1].c-e*T[0].c),1/r[1]*(T[0].a-e*T[1].a),1/r[1]*(T[0].b-e*T[1].b),1/r[1]*(T[0].c-e*T[1].c)]),A[3*this.a]+=y*(P[2]*D[1].b-P[1]*D[1].c),A[3*this.a+1]+=y*(P[0]*D[1].c-P[2]*D[1].a),A[3*this.a+2]+=y*(P[1]*D[1].a-P[0]*D[1].b),A[3*this.b]+=y*(P[1]*(D[1].c-D[0].c)+P[2]*(D[0].b-D[1].b)+P[4]*-D[3].c+P[5]*D[3].b),A[3*this.b+1]+=y*(P[0]*(D[0].c-D[1].c)+P[2]*(D[1].a-D[0].a)+P[3]*D[3].c+P[5]*-D[3].a),A[3*this.b+2]+=y*(P[0]*(D[1].b-D[0].b)+P[1]*(D[0].a-D[1].a)+P[3]*-D[3].b+P[4]*D[3].a),A[3*this.c]+=y*(P[1]*D[0].c+P[2]*-D[0].b+P[4]*(D[3].c-D[2].c)+P[5]*(D[2].b-D[3].b)),A[3*this.c+1]+=y*(P[0]*-D[0].c+P[2]*D[0].a+P[3]*(D[2].c-D[3].c)+P[5]*(D[3].a-D[2].a)),A[3*this.c+2]+=y*(P[0]*D[0].b+P[1]*-D[0].a+P[3]*(D[3].b-D[2].b)+P[4]*(D[2].a-D[3].a)),A[3*this.d]+=y*(P[4]*D[2].c-P[5]*D[2].b),A[3*this.d+1]+=y*(P[5]*D[2].a-P[3]*D[2].c),A[3*this.d+2]+=y*(P[3]*D[2].b-P[4]*D[2].a));},s.a=0,s.b=0,s.c=0,s.d=0,s.e=0,s.f=0,s.g=0,o.D9=N(363),Z.xD=function(t,A){var e,r,P,i,D;return i=rt.kF(A.p,t.b),D=rt.kF(A.p,t.d),e=rt.gF(A.p,t.b),r=rt.gF(A.p,t.d),P=(i-D)/(i+D),.5*(i+D)*(1+(e==68||r==68?0:.2*(1-d.Math.exp(-12*P*P))));},Z.yD=function(t,A,e){var r,P,i,D,u,Q,l;return i=rt.hF(A.p,t.b),D=rt.hF(A.p,t.d),r=rt.fF(A.p,t.b),P=rt.fF(A.p,t.d),u=rt.jF(A.p,t.b),Q=rt.jF(A.p,t.d),l=e*e,181.16*i*D*r*P/((d.Math.sqrt(r/u)+d.Math.sqrt(P/Q))*l*l*l);},Z.zD=function(t,A,e,r){var P,i;this.g=A,this.b=A.a[e],this.d=A.a[r],this.a=e,this.c=r,P=Z.xD(this,t),i=Z.yD(this,t,P),this.e=rt.gF(t.p,this.b),this.f=rt.gF(t.p,this.d),(this.e==68&&this.f==65||this.e==65&&this.f==68)&&(P=P*.8,i*=.5),this.i=P,this.j=i;},Z.AD=function(t,A,e,r){var P,i,D,u,Q;for(Q=new U.Mj(),i=0;i<A.q;i++)for(D=i+1;D<A.q;D++)P=new Z.iD(i,D),u=U.nob(e.a,P)!=null?U.nob(e.a,P):(Z.eD(),Z.dD),(u==(Z.eD(),Z._C)||u==Z.dD)&&Z.HD(new Z.MD(A,i,D))<r&&U.sj(Q,new Z.zD(t,A,i,D));return Q;},x(364,1,Un,Z.zD),s.cc=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c;return r&&(n.Tl(this.g,this.a)==1||n.Tl(this.g,this.c)==1)?0:(Q=Z.HD(new Z.PD(t,this.a,this.c)),l=Q*Q,L=l*l*l*Q,P=1.07*this.i/(Q+y9*this.i),i=P*P,D=i*i*i*P,T=this.i*this.i,c=T*T*T*this.i,u=1.12*c/(L+T9*c)-2,y=D*u*this.j,y);},s.dc=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c;for(P=Z.HD(new Z.PD(t,this.a,this.c)),D=P/this.i,u=D*D,Q=u*u*u,l=Q*D,L=l+T9,y=1.07/(D+1.07-1),T=y*y,c=T*T*T*y,e=this.j/this.i*c*(h9*Q/(L*L)+(h9/L+14)/(D+y9)),i=0;i<3;i++)r=.01*this.i,P>0&&(r=e*(t[3*this.a+i]-t[3*this.c+i])/P),A[3*this.a+i]+=r,A[3*this.c+i]-=r;},s.a=0,s.b=0,s.c=0,s.d=0,s.e=0,s.f=0,s.i=0,s.j=0,o.E9=N(364),Z.DD=function(t,A){return d.Math.acos((t.a*A.a+t.b*A.b+t.c*A.c)/(d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)*d.Math.sqrt(A.a*A.a+A.b*A.b+A.c*A.c)));},Z.ED=function(t,A){return new Z.KD(t.b*A.c-t.c*A.b,t.c*A.a-t.a*A.c,t.a*A.b-t.b*A.a);},Z.FD=function(t,A){return d.Math.sqrt((t.a-A.a)*(t.a-A.a)+(t.b-A.b)*(t.b-A.b)+(t.c-A.c)*(t.c-A.c));},Z.GD=function(t,A){return t.a*A.a+t.b*A.b+t.c*A.c;},Z.HD=function(t){return d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c);},Z.ID=function(t){return d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)>0?new Z.KD(t.a/d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),t.b/d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c),t.c/d.Math.sqrt(t.a*t.a+t.b*t.b+t.c*t.c)):new Z.KD(0,0,0);},Z.JD=function(t,A){return new Z.KD(A.a-t.a,A.b-t.b,A.c-t.c);},Z.KD=function(t,A,e){this.a=t,this.b=A,this.c=e;},Z.LD=function(t,A){this.a=t.J[A].a,this.b=t.J[A].b,this.c=t.J[A].c;},Z.MD=function(t,A,e){Z.ND.call(this,Z.JD(new Z.LD(t,A),new Z.LD(t,e)));},Z.ND=function(t){this.a=t.a,this.b=t.b,this.c=t.c;},Z.OD=function(t,A){this.a=t[3*A],this.b=t[3*A+1],this.c=t[3*A+2];},Z.PD=function(t,A,e){Z.ND.call(this,Z.JD(new Z.OD(t,A),new Z.OD(t,e)));},x(22,1,{22:1},Z.KD,Z.LD,Z.MD,Z.OD,Z.PD),s.mb=function(t){var A;return t===this?true:Zt(t,22)?(A=t,d.Math.abs(this.a-A.a)<.01&&d.Math.abs(this.b-A.b)<.01&&d.Math.abs(this.c-A.c)<.01):false;},s.ob=function(){return Mt(this.a)+Mt(this.b)+Mt(this.c);},s.a=0,s.b=0,s.c=0,o.F9=N(22),rt.SD=function(t,A,e,r,P,i){var D,u,Q,l,L,y,T,c,b;for(b=j(F(o.O7,1),Ft,6,15,[0,0,0]),l=j(F(o.O7,1),Ft,6,15,[0,0,0]),u=j(F(o.Q7,1),H,6,15,[A.C[e],A.C[r],A.C[P]]),Q=1.75,L=0;L<3;L++)switch(u[L]){case 1:b[L]=1.395;break;case 6:b[L]=2.494,l[L]=1.016;break;case 7:b[L]=2.711,l[L]=1.113;break;case 8:b[L]=3.045,l[L]=1.337;break;case 9:b[L]=2.847;break;case 14:b[L]=2.35,l[L]=.811;break;case 15:b[L]=2.35,l[L]=1.068;break;case 16:b[L]=2.98,l[L]=1.249;break;case 17:b[L]=2.909,l[L]=1.078;break;case 35:b[L]=3.017;break;case 53:b[L]=3.086;}return y=rt.pE(t.a.d,A,e,r),T=rt.pE(t.a.d,A,r,P),D=(y-T)*(y-T)/((y+T)*(y+T)),c=e9*i,P1.qF(A,e,r,P,4)?Q*=.85:P1.qF(A,e,r,P,3)&&(Q*=.05),Q*b[0]*l[1]*b[2]/((y+T)*c*c*d.Math.exp(2*D));},rt.TD=function(t,A,e,r,P){var i;if(P1.qF(A,e,r,P,3))return 60;if(P1.qF(A,e,r,P,4))return 90;switch(i=A.a[r],rt._D(t.a.b,i)){case 2:if(A.C[r]==8)return 105;if(rt.bE(t.a.b,i))return 180;break;case 3:if(rt.fE(t.a.b,i)==3&&rt.cE(t.a.b,i)==0)return A.C[r]==7?107:92;break;case 4:return 109.45;}return 120;},rt.UD=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c;for(i=A.a[e],u=A.a[r],Q=A.a[P],L=P1.pF(t.a,A,e,r,P),T=-1,y=0;y<5&&T<0;y++)D=t.a.g.a[i-1][y],l=t.a.g.a[Q-1][y],D>l&&(D=(c=h.qjb(l),h.qjb(l=D),c).a),T=Z.XC(j(F(o.Q7,1),H,6,15,[2,1,3,0]),j(F(o.Q7,1),H,6,15,[u,D,l,L]),t);return T;},rt.VD=function(t,A,e,r,P){var i;return i=rt.UD(t,A,e,r,P),i<0?rt.SD(t,A,e,r,P,rt.TD(t,A,e,r,P)):d.Math.abs(h.Nib(t.b[i][4]))<.001?rt.SD(t,A,e,r,P,h.Nib(t.b[i][5])):h.Nib(t.b[i][4]);},rt.WD=function(t,A,e,r,P){var i;return i=rt.UD(t,A,e,r,P),i<0?rt.TD(t,A,e,r,P):h.Nib(t.b[i][5]);},rt.XD=function(t){this.b=Z.nC("/resources/forcefield/mmff94/angle.csv"),this.a=t;},x(385,1,{},rt.XD),s.kc=function(t,A){return h.Oib(this.b[t][A]);},s.lc=function(){return this.b.length;},o.G9=N(385),rt.$D=function(t,A){return t.a[Z.VC(0,A,0,t.a.length,true,t)][6]>0;},rt._D=function(t,A){return t.a[Z.VC(0,A,0,t.a.length,true,t)][2];},rt.aE=function(t){return t.a.length;},rt.bE=function(t,A){var e;return e=Z.VC(0,A,0,t.a.length,true,t),e>=0&&t.a[e][7]>0;},rt.cE=function(t,A){return t.a[Z.VC(0,A,0,t.a.length,true,t)][5];},rt.dE=function(t,A){return t.a[Z.VC(0,A,0,t.a.length,true,t)][4];},rt.eE=function(t,A){return t.a[Z.VC(0,A,0,t.a.length,true,t)][8]>0;},rt.fE=function(t,A){return t.a[Z.VC(0,A,0,t.a.length,true,t)][3];},rt.gE=function(){this.a=Z.oC("/resources/forcefield/mmff94/atom.csv");},x(379,1,{},rt.gE),s.kc=function(t,A){return this.a[t][A];},s.lc=function(){return rt.aE(this);},o.H9=N(379),rt.jE=function(t,A){return h.Nib(t.a[A][3]);},rt.kE=function(t,A){return h.Nib(t.a[A][2]);},rt.lE=function(){this.a=Z.nC("/resources/forcefield/mmff94/bndk.csv");},x(392,1,{},rt.lE),s.kc=function(t,A){return h.Oib(this.a[t][A]);},s.lc=function(){return this.a.length;},o.I9=N(392),rt.oE=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g;return L=P1.AF(t.a,A,n.Xo(A,e,r)),i=A.a[e],u=A.a[r],i>u&&(i=(b=h.qjb(u),h.qjb(u=i),b).a),T=Z.XC(j(F(o.Q7,1),H,6,15,[1,2,0]),j(F(o.Q7,1),H,6,15,[i,u,L]),t),T>=0?h.Nib(t.b[T][3]):(P=A.C[e],D=A.C[r],P>D&&(P=(c=h.qjb(D),h.qjb(D=P),c).a),g=rt.pE(t,A,e,r),l=Z.XC(j(F(o.Q7,1),H,6,15,[0,1]),j(F(o.Q7,1),H,6,15,[P,D]),t.a.c),l>=0?(y=d.Math.pow(rt.kE(t.a.c,l)/g,6),rt.jE(t.a.c,l)*y):(Q=Z.XC(j(F(o.Q7,1),H,6,15,[0,1]),j(F(o.Q7,1),H,6,15,[Z.IC(P),Z.IC(D)]),t.a.j),d.Math.pow(10,-(g-rt.ME(t.a.j,Q,2))/rt.ME(t.a.j,Q,3))));},rt.pE=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;return L=P1.AF(t.a,A,n.Xo(A,e,r)),D=A.a[e],l=A.a[r],D>l&&(D=(k=h.qjb(l),h.qjb(l=D),k).a),b=Z.XC(j(F(o.Q7,1),H,6,15,[1,2,0]),j(F(o.Q7,1),H,6,15,[D,l,L]),t),b>=0?h.Nib(t.b[b][4]):(i=A.C[e],Q=A.C[r],i>Q&&(i=(g=h.qjb(Q),h.qjb(Q=i),g).a),P=Z.WC(i,t.a.f),u=Z.WC(Q,t.a.f),v=rt.CE(t.a.f,P),S=rt.CE(t.a.f,u),T=rt.AE(t.a.f,P),c=rt.AE(t.a.f,u),y=i==1||Q==1?.05:.085,v+S-y*d.Math.pow(d.Math.abs(T-c),1.4));},rt.qE=function(t){this.b=Z.nC("/resources/forcefield/mmff94/bond.csv"),this.a=t;},x(388,1,{},rt.qE),s.kc=function(t,A){return h.Oib(this.b[t][A]);},s.lc=function(){return this.b.length;},o.J9=N(388),rt.tE=function(t,A,e,r){return Z.VC(1,A,0,e,r,new rt.xE(t));},rt.uE=function(t,A){return h.Nib(t.b[A-1][2]);},rt.vE=function(t,A,e,r){var P,i,D,u,Q,l,L;return L=e>r?1:-1,l=e>r?r:e,Q=e>r?e:r,i=rt.tE(t,l,t.a.length,true),P=rt.tE(t,l,t.a.length,false),i==-1||P==-1||(u=Z.VC(2,Q,i,P+1,true,new rt.xE(t)),D=Z.VC(2,Q,i,P+1,false,new rt.xE(t)),u==-1||D==-1)?h.Nib(t.b[e-1][1])-h.Nib(t.b[r-1][1]):A==0&&h.Oib(t.a[u][0])==0?L*h.Nib(t.a[u][3]):A==1&&h.Oib(t.a[D][0])==1?L*h.Nib(t.a[D][3]):h.Nib(t.b[e-1][1])-h.Nib(t.b[r-1][1]);},rt.wE=function(){this.b=Z.nC("/resources/forcefield/mmff94/pbci.csv"),this.a=Z.nC("/resources/forcefield/mmff94/bci.csv");},x(386,1,{},rt.wE),o.L9=N(386),rt.xE=function(t){this.a=t;},x(196,1,{},rt.xE),s.kc=function(t,A){return h.Oib(this.a.a[t][A]);},s.lc=function(){return this.a.a.length;},o.K9=N(196),rt.AE=function(t,A){return h.Nib(t.a[A][2]);},rt.BE=function(t){return t.a.length;},rt.CE=function(t,A){return h.Nib(t.a[A][1]);},rt.DE=function(){this.a=Z.nC("/resources/forcefield/mmff94/covrad.csv");},x(393,1,{},rt.DE),s.kc=function(t,A){return h.Oib(this.a[t][A]);},s.lc=function(){return rt.BE(this);},o.M9=N(393),rt.GE=function(){this.a=Z.oC("/resources/forcefield/mmff94/def.csv");},x(395,1,{},rt.GE),o.N9=N(395),rt.HE=function(t,A,e,r,P){var i,D,u,Q;return i=Z.HC(A.C[e]),D=Z.HC(A.C[r]),u=Z.HC(A.C[P]),i>u&&(i=(Q=h.qjb(u),h.qjb(u=i),Q).a),Z.XC(j(F(o.Q7,1),H,6,15,[1,0,2]),j(F(o.Q7,1),H,6,15,[D,i,u]),t);},rt.IE=function(t,A,e,r,P){var i,D,u,Q;return i=Z.HC(A.C[e]),D=Z.HC(A.C[P]),u=i>D?4:3,Q=rt.HE(t,A,e,r,P),Q>=0?h.Nib(t.a[Q][u]):0;},rt.JE=function(){this.a=Z.nC("/resources/forcefield/mmff94/dfsb.csv");},x(394,1,{},rt.JE),s.kc=function(t,A){return h.Oib(this.a[t][A]);},s.lc=function(){return this.a.length;},o.O9=N(394),rt.ME=function(t,A,e){return h.Nib(t.a[A][e]);},rt.NE=function(){this.a=Z.nC("/resources/forcefield/mmff94/herschbachlaurie.csv");},x(396,1,{},rt.NE),s.kc=function(t,A){return h.Oib(this.a[t][A]);},s.lc=function(){return this.a.length;},o.P9=N(396),rt.QE=function(t,A,e,r,P,i){var D,u,Q,l,L,y;for(D=A.a[e],L=j(F(o.Q7,1),H,6,15,[A.a[r],A.a[P],A.a[i]]),u=0;u<4;u++){for(y=B(o.Q7,H,6,3,15,1),l=0;l<3;l++)y[l]=t.a.g.a[L[l]-1][u+1];if(w.lAb(y,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),Q=Z.XC(j(F(o.Q7,1),H,6,15,[1,0,2,3]),j(F(o.Q7,1),H,6,15,[D,y[0],y[1],y[2]]),t),Q>=0)return h.Nib(t.b[Q][4]);}return 0;},rt.RE=function(t,A){this.b=Z.nC(A),this.a=t;},x(387,1,{},rt.RE),s.kc=function(t,A){return h.Oib(this.b[t][A]);},s.lc=function(){return this.b.length;},o.Q9=N(387),rt.UE=function(t,A,e,r,P){var i,D,u,Q,l;return i=A.a[e],D=A.a[r],u=A.a[P],Q=P1.oF(t.a,A,e,r,P),i>u&&(i=(l=h.qjb(u),h.qjb(u=i),l).a),Z.XC(j(F(o.Q7,1),H,6,15,[2,1,3,0]),j(F(o.Q7,1),H,6,15,[D,i,u,Q]),t);},rt.VE=function(t,A,e,r,P){var i,D,u,Q,l,L;return i=A.a[e],D=A.a[P],Q=P1.BF(t.a,A,e,r),l=P1.BF(t.a,A,r,P),L=rt.UE(t,A,e,r,P),u=i>D||i==D&&Q<l?1:0,L>=0?h.Nib(t.b[L][4+u]):rt.IE(t.a.i,A,e,r,P);},rt.WE=function(t){this.b=Z.nC("/resources/forcefield/mmff94/stbn.csv"),this.a=t;},x(389,1,{},rt.WE),s.kc=function(t,A){return h.Oib(this.b[t][A]);},s.lc=function(){return this.b.length;},o.R9=N(389),rt.ZE=function(t,A,e,r,P,i){var D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot;for(L=A.a[e],y=A.a[r],T=A.a[P],c=A.a[i],ut=P1.EF(t.a,A,e,r,P,i),Dt=ut>10?ut/10|0:ut,ot=ut>10?ut-Dt*10:0,K=-1,O=0,G=0,tt=0,q=Dt,it=5;(O<it&&(K==-1||it==4)||O==4&&Dt==5&&ot>0)&&(it==5&&O==4&&(it=4,O=0,q=ot),O==1?(G=1,tt=3):O==2?(G=3,tt=1):(G=O,tt=O),v=t.a.g.a[L-1][d.Math.min(G+1,4)],S=y,p=T,E=t.a.g.a[c-1][d.Math.min(tt+1,4)],S>p?(p=(z=h.qjb(S),h.qjb(S=p),z).a,E=(Qt=h.qjb(v),h.qjb(v=E),Qt).a):S==p&&v>E&&(E=(Qt=h.qjb(v),h.qjb(v=E),Qt).a),K=Z.XC(j(F(o.Q7,1),H,6,15,[2,3,1,4,0]),j(F(o.Q7,1),H,6,15,[S,p,v,E,q]),t),!(K!=-1&&it==4));)++O;if(K>=0)return new rt.dF(t,K);for(k=n.Xo(A,r,P),u=j(F(o.O7,1),Ft,6,15,[0,0]),Q=j(F(o.O7,1),Ft,6,15,[0,0]),l=j(F(o.O7,1),Ft,6,15,[0,0]),b=j(F(o.Q7,1),H,6,15,[A.C[r],A.C[P]]),D=(rt._D(t.a.b,y)-1)*(rt._D(t.a.b,T)-1),V=0;V<2;V++)switch(b[V]){case 6:u[V]=2,Q[V]=2.12;break;case 7:u[V]=2,Q[V]=1.5;break;case 8:u[V]=2,Q[V]=.2,l[V]=2;break;case 14:u[V]=1.25,Q[V]=1.22;break;case 15:u[V]=1.25,Q[V]=2.4;break;case 16:u[V]=1.25,Q[V]=.49,l[V]=8;}return rt.bE(t.a.b,y)||rt.bE(t.a.b,T)?new rt.cF(0,0):rt.$D(t.a.b,y)&&rt.$D(t.a.b,T)&&k<A.g&&n.pt(A.p,k)?(g=rt.fE(t.a.b,y)==3&&rt.fE(t.a.b,T)==4||rt.fE(t.a.b,y)==4&&rt.fE(t.a.b,T)==3?3:6,lt=rt.dE(t.a.b,y)==0&&rt.dE(t.a.b,T)==0?.5:.3,new rt.cF(g*lt*d.Math.sqrt(u[0]*u[1]),0)):n.fm(A,k)==2?(lt=rt.cE(t.a.b,y)==2&&rt.cE(t.a.b,T)==2?1:.4,new rt.cF(6*lt*d.Math.sqrt(u[0]*u[1]),0)):rt._D(t.a.b,y)==4&&rt._D(t.a.b,T)==4?new rt.cF(0,d.Math.sqrt(Q[0]*Q[1])/D):rt._D(t.a.b,y)==4&&rt._D(t.a.b,T)!=4?rt._D(t.a.b,T)==3&&(rt.fE(t.a.b,T)==4||rt.fE(t.a.b,T)==34||rt.cE(t.a.b,T)>0)||rt._D(t.a.b,T)==2&&(rt.fE(t.a.b,T)==3||rt.cE(t.a.b,T)>0)?new rt.bF():new rt.cF(0,d.Math.sqrt(Q[0]*Q[1])/D):rt._D(t.a.b,T)==4&&rt._D(t.a.b,y)!=4?rt._D(t.a.b,y)==3&&(rt.fE(t.a.b,y)==4||rt.fE(t.a.b,y)==34||rt.cE(t.a.b,y)>0)||rt._D(t.a.b,y)==2&&(rt.fE(t.a.b,y)==3||rt.cE(t.a.b,y)>0)?new rt.bF():new rt.cF(0,d.Math.sqrt(Q[0]*Q[1])/D):n.fm(A,k)==1&&rt.cE(t.a.b,y)>0&&rt.cE(t.a.b,T)>0||rt.cE(t.a.b,y)>0&&rt.dE(t.a.b,T)>0||rt.dE(t.a.b,y)>0&&rt.cE(t.a.b,T)>0?rt.dE(t.a.b,y)>0&&rt.dE(t.a.b,T)>0?new rt.bF():rt.dE(t.a.b,y)>0&&rt.cE(t.a.b,T)>0?(lt=0,rt.cE(t.a.b,y)==1?lt=.5:Z.HC(b[0])==2&&Z.HC(b[1])==2?lt=.3:(Z.HC(b[0])!=2||Z.HC(b[1])!=2)&&(lt=.15),new rt.cF(6*lt*d.Math.sqrt(u[0]*u[1]),0)):rt.dE(t.a.b,T)>0&&rt.cE(t.a.b,y)>0?(lt=0,rt.cE(t.a.b,T)==1?lt=.5:Z.HC(b[0])==2&&Z.HC(b[1])==2?lt=.3:(Z.HC(b[0])!=2||Z.HC(b[1])!=2)&&(lt=.15),new rt.cF(6*lt*d.Math.sqrt(u[0]*u[1]),0)):(rt.cE(t.a.b,y)==1||rt.cE(t.a.b,T)==1)&&(b[0]!=6||b[1]!=6)?new rt.cF(2.4000000000000004*d.Math.sqrt(u[0]*u[1]),0):new rt.cF(.8999999999999999*d.Math.sqrt(u[0]*u[1]),0):(b[0]==8||b[0]==16)&&(b[1]==8||b[1]==16)?new rt.cF(-d.Math.sqrt(l[0]*l[1]),0):new rt.cF(0,d.Math.sqrt(Q[0]*Q[1])/D);},rt.$E=function(t,A){this.b=Z.nC(A),this.a=t;},x(390,1,{},rt.$E),s.kc=function(t,A){return h.Oib(this.b[t][A]);},s.lc=function(){return this.b.length;},o.T9=N(390),rt.bF=function(){this.a=0,this.b=0,this.c=0;},rt.cF=function(t,A){this.a=0,this.b=t,this.c=A;},rt.dF=function(t,A){this.a=h.Nib(t.b[A][5]),this.b=h.Nib(t.b[A][6]),this.c=h.Nib(t.b[A][7]);},x(49,1,{},rt.bF,rt.cF,rt.dF),s.pb=function(){return this.a+","+this.b+","+this.c;},s.a=0,s.b=0,s.c=0,o.S9=N(49),rt.fF=function(t,A){return h.Nib(t.a[Z.VC(0,A,0,t.a.length,true,t)][1]);},rt.gF=function(t,A){return t.a[Z.VC(0,A,0,t.a.length,true,t)][5].a;},rt.hF=function(t,A){return h.Nib(t.a[Z.VC(0,A,0,t.a.length,true,t)][4]);},rt.iF=function(t){return t.a.length;},rt.jF=function(t,A){return h.Nib(t.a[Z.VC(0,A,0,t.a.length,true,t)][2]);},rt.kF=function(t,A){var e;return e=Z.VC(0,A,0,t.a.length,true,t),h.Nib(t.a[e][3])*d.Math.pow(h.Nib(t.a[e][1]),.25);},rt.lF=function(){this.a=Z.nC("/resources/forcefield/mmff94/vanderwaals.csv");},x(391,1,{},rt.lF),s.kc=function(t,A){return h.Oib(this.a[t][A]);},s.lc=function(){return rt.iF(this);},o.U9=N(391),P1.oF=function(t,A,e,r,P){var i,D,u,Q,l,L,y;switch(i=A.a[e],D=A.a[P],l=P1.BF(t,A,e,r),y=P1.BF(t,A,r,P),Q=i<=D?l:y,L=i<D?y:l,u=P1.pF(t,A,e,r,P),u){case 1:return Q>0||Q==L?1:2;case 2:return 3;case 3:return 5;case 4:return 4;case 5:return Q>0||Q==L?6:7;case 6:return 8;case 7:return Q>0||Q==L?9:10;case 8:return 11;}return 0;},P1.pF=function(t,A,e,r,P){var i,D;return i=P1.BF(t,A,e,r)+P1.BF(t,A,r,P),D=i,P1.qF(A,e,r,P,3)?D+=i>0?4:3:P1.qF(A,e,r,P,4)&&(D+=i>0?6:4),D;},P1.qF=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T;if(T=(n.bw(t,7),t.p),l=new U.rub(),U.qub(l,h.qjb(A)),U.qub(l,h.qjb(e)),U.qub(l,h.qjb(r)),n.Xo(t,A,e)>=0&&n.Xo(t,e,r)>=0){for(L=0;L<T.j.a.length;L++)if(U.xj(T.k,L).length==P){for(y=new U.rub(),D=U.xj(T.j,L),u=0,Q=D.length;u<Q;++u)i=D[u],U.qub(y,h.qjb(i));if(U.Ji(y,l))return  true;}}return  false;},P1.rF=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt,c1,D1,St,gt,w1,I1,L1,G1,s1,Q1,z1,t0,p0,J1,u1,u0,h0,d0,RA,BA,nA,eA,R0,CA,K0,C0,YA,dn,vt,Wt,kA,mn,vn,Fn;if(A<t.f&&n.ot(t.p,A)){if(P1.yF(t,A,5)){if(g=new U.Nj(t.q),S=new U.Nj(t.q),b=false,tt=false,Qt=false,t.C[A]==6||t.C[A]==7){for(G=0;G<t.e[A];G++)if(vt=t.i[A][G],!!P1.yF(t,vt,5))for(P1.wF(t,A,vt)&&(t.C[vt]==8||t.C[vt]==16||t.C[vt]==7&&t.e[vt]+n.op(t,vt)==3&&!P1.xF(t,vt))&&U.sj(g,h.qjb(vt)),z1=0;z1<t.e[vt];z1++)Wt=t.i[vt][z1],Wt!=A&&P1.yF(t,Wt,5)&&P1.wF(t,A,Wt)&&(t.C[Wt]==8||t.C[Wt]==16||t.C[Wt]==7&&t.e[Wt]+n.op(t,Wt)==3&&!P1.xF(t,Wt))&&U.sj(S,h.qjb(Wt));for(c=new U.rrb(g);c.a<c.c.a.length;)if(T=U.qrb(c).a,t.C[T]==8||t.C[T]==16){tt=true;break;}for(v=new U.rrb(S);v.a<v.c.a.length;)if(k=U.qrb(v).a,t.C[k]==8||t.C[k]==16){Qt=true;break;}for(V=0;V<g.a.length;V++)for(Q1=V;Q1<S.a.length;Q1++)if(P1.wF(t,(w.FAb(V,g.a.length),g.a[V]).a,(w.FAb(Q1,S.a.length),S.a[Q1]).a)){b=true;break;}}switch(t.C[A]){case 6:if(S.a.length==0){for(h0=0,p0=0,u1=0,u0=0,V=0;V<t.e[A];V++)vt=t.i[A][V],t.C[vt]==7&&t.e[vt]+n.op(t,vt)==3&&(++h0,t.s[vt]>0&&!P1.xF(t,vt)&&++p0,P1.yF(t,vt,5)&&++u1,P1.yF(t,vt,6)&&++u0);if((h0==2&&u1>0||h0==3&&u1==2)&&p0>0&&u0==0)return 80;}if(!(g.a.length==0^S.a.length==0)){for(Fn=true,vn=true,V=0;V<t.e[A];V++)vt=t.i[A][V],(t.C[vt]!=6||!P1.uF(t,vt))&&(Fn=false),P1.wF(t,A,vt)&&!(vt<t.f&&n.ot(t.p,vt))&&(vn=false);if(g.a.length==0&&S.a.length==0&&!Fn&&vn||g.a.length!=0&&g.a.length!=0&&(!b||!tt&&!Qt))return 78;}if(g.a.length!=0&&(S.a.length==0||tt))return 63;if(S.a.length!=0&&(g.a.length==0||Qt))return 64;break;case 7:if(P1.xF(t,A))return 82;if(g.a.length==0&&S.a.length==0)return t.e[A]+n.op(t,A)==3?39:76;if(t.e[A]+n.op(t,A)==3&&g.a.length==0^S.a.length==0)return 81;if(g.a.length!=0&&(S.a.length==0||tt))return 65;if(S.a.length!=0&&(g.a.length==0||Qt))return 66;if(g.a.length!=0&&S.a.length!=0)return 79;break;case 8:return 59;case 16:return 44;}}if(P1.yF(t,A,6))switch(t.C[A]){case 6:return 37;case 7:return P1.xF(t,A)?69:t.e[A]+n.op(t,A)==3?58:38;}}switch(t.C[A]){case 3:if(t.j[A]==0)return 92;break;case 6:if(t.e[A]+n.op(t,A)==4)return (t.p&&A<t.f?n.dt(t.p,A):0)==3?22:(t.p&&A<t.f?n.dt(t.p,A):0)==4?20:1;if(t.e[A]+n.op(t,A)==3){for(d0=0,RA=0,nA=0,K0=0,E=0,G=0;G<t.e[A];G++)vt=t.i[A][G],n.fm(t,n.Xo(t,A,vt))==2&&(E=t.C[vt]),t.e[vt]+n.op(t,vt)==1?(t.C[vt]==8&&++nA,t.C[vt]==16&&++K0):t.C[vt]==7&&(t.e[vt]+n.op(t,vt)==3?++RA:t.e[vt]+n.op(t,vt)==2&&n.fm(t,n.Xo(t,A,vt))==2&&++d0);return RA>=2&&d0==0&&E==7?57:nA==2||K0==2?41:(t.p&&A<t.f?n.dt(t.p,A):0)==4&&E==6?30:E==7||E==8||E==15||E==16?3:2;}if(t.e[A]+n.op(t,A)==2)return 4;if(t.e[A]+n.op(t,A)==1)return 60;break;case 7:for(YA=0,L1=false,K=0;K<t.e[A];K++)if(vt=t.i[A][K],t.C[vt]==8&&t.e[vt]+n.op(t,vt)==1&&++YA,n.vp(t,A)+n.op(t,A)>=3&&(t.C[vt]==15||t.C[vt]==16)){for(CA=0,Q1=0;Q1<t.e[vt];Q1++)Wt=t.i[vt][Q1],t.C[Wt]==8&&t.e[Wt]+n.op(t,Wt)==1&&++CA;L1||(L1=CA>=2);}if(t.e[A]+n.op(t,A)==4)return P1.xF(t,A)?68:34;if(t.e[A]+n.op(t,A)==3){if(n.vp(t,A)+n.op(t,A)>=4){for(q=false,V=0;V<t.e[A];V++)if(vt=t.i[A][V],n.fm(t,n.Xo(t,A,vt))==2&&(q=t.C[vt]==7||t.C[vt]==6,t.C[vt]==6))for(Q1=0;q&&Q1<t.e[vt];Q1++)Wt=t.i[vt][Q1],Wt!=A&&(q=!(t.C[Wt]==7&&t.e[Wt]+n.op(t,Wt)==3));if(YA==1)return 67;if(YA>=2)return 45;if(q)return 54;}if(n.vp(t,A)+n.op(t,A)>=3){for(Ut=false,bt=false,wt=false,ct=false,xt=false,Kt=false,e=0,r=0,P=0,i=0,u=0,L=0,V=0;V<t.e[A];V++)if(vt=t.i[A][V],t.C[vt]==6){for(xt=true,vt<t.f&&n.ot(t.p,vt)&&(t.p&&vt<t.f?n.dt(t.p,vt):0)==6&&(Kt=true),P=0,i=0,u=0,L=0,p0=0,O=0,Q1=0;Q1<t.e[vt];Q1++){if(Wt=t.i[vt][Q1],p=n.Xo(t,vt,Wt),n.fm(t,p)==2&&(t.C[Wt]==8||t.C[Wt]==16)&&(Ut=true),(n.fm(t,p)==2||p<t.g&&n.pt(t.p,p)&&(t.C[Wt]==6||t.C[Wt]==7&&P1.vF(t,Wt)==1))&&(e=t.C[Wt]),n.fm(t,p)==3&&(r=t.C[Wt]),t.C[Wt]==7&&t.e[Wt]+n.op(t,Wt)==3){for(t.s[Wt]==1&&++p0,P1.yF(t,vt,6)&&++O,l=0,t0=0;t0<t.e[Wt];t0++)kA=t.i[Wt][t0],t.C[kA]==8&&++l;l<2&&++i;}t.C[Wt]==7&&t.e[Wt]+n.op(t,Wt)==2&&(n.fm(t,p)==2||p<t.g&&n.pt(t.p,p))&&++P,Wt<t.f&&n.ot(t.p,Wt)&&(t.C[Wt]==8&&++u,t.C[Wt]==16&&++L);}e==7&&(i==2&&P==0&&p0>0&&O==0&&t.e[vt]+n.op(t,vt)<4&&(bt=true),i==3&&(wt=true));}else if(t.C[vt]==7){for(D=0,Q=0,y=0,Q1=0;Q1<t.e[vt];Q1++)if(Wt=t.i[vt][Q1],p=n.Xo(t,vt,Wt),n.fm(t,p)==2){if(t.C[Wt]==6){for(t0=0;t0<t.e[Wt];t0++)kA=t.i[Wt][t0],kA!=A&&(t.C[kA]==7?++D:t.C[kA]==8?++Q:t.C[kA]==16&&++y);D==0&&Q==0&&y==0&&!Kt&&(ct=true);}t.C[Wt]==7&&!Kt&&(ct=true);}}if(xt){if(r==7&&(L1=true),bt)return 55;if(wt)return 56;if(!Ut&&!L1&&(u==0&&L==0&&Kt||e==6||e==7||e==15||r==6))return 40;}if(!L1&&(Ut||ct))return 10;}}if(t.e[A]+n.op(t,A)==2){if(n.vp(t,A)+n.op(t,A)==4){for(Dt=false,V=0;!Dt&&V<t.e[A];V++)vt=t.i[A][V],Dt=n.fm(t,n.Xo(t,A,vt))==3;return Dt?61:53;}if(n.vp(t,A)+n.op(t,A)==3){for(kt=false,ut=false,V=0;V<t.e[A];V++)vt=t.i[A][V],n.fm(t,n.Xo(t,A,vt))==2&&(kt=t.C[vt]==8&&YA==1,ut=t.C[vt]==6||t.C[vt]==7);if(kt&&!ut)return 46;if(ut)return 9;}if(n.vp(t,A)+n.op(t,A)>=2){for(Et=false,V=0;V<t.e[A];V++)if(vt=t.i[A][V],t.C[vt]==16){for(dn=0,Q1=0;Q1<t.e[vt];Q1++)Wt=t.i[vt][Q1],t.C[Wt]==8&&t.e[Wt]+n.op(t,Wt)==1&&++dn;Et=dn==1;}if(Et)return 48;if(!L1)return 62;}}if(L1)return 43;if(t.e[A]+n.op(t,A)==1){for(Ht=false,ot=false,V=0;!Ht&&!ot&&V<t.e[A];V++)if(vt=t.i[A][V],Ht=n.fm(t,n.Xo(t,A,vt))==3,t.C[vt]==7&&t.e[vt]+n.op(t,vt)==2)for(Q1=0;!ot&&Q1<t.e[vt];Q1++)Wt=t.i[vt][Q1],ot=t.C[Wt]==7&&t.e[Wt]+n.op(t,Wt)==2||t.C[Wt]==6&&t.e[Wt]+n.op(t,Wt)==3;if(Ht)return 42;if(ot)return 47;}return 8;case 8:if(t.e[A]+n.op(t,A)==3)return 49;if(t.e[A]+n.op(t,A)==2){if(n.vp(t,A)+n.op(t,A)==3)return 51;for(J1=0,V=0;V<t.e[A];V++)vt=t.i[A][V],t.C[vt]==1&&++J1;return J1+n.op(t,A)==2?70:6;}if(t.j[A]<=1){for(BA=0,R0=0,C0=0,St=t.e[A]-t.j[A]+n.op(t,A)>0,it=false,z=false,D1=false,Bt=false,gt=false,It=false,Nt=false,s1=false,I1=false,G1=false,w1=false,V=0;V<t.e[A]&&!(St||it||z||D1||Bt||gt||It||Nt||s1||I1||G1||w1);V++){if(vt=t.i[A][V],t.C[vt]==6||t.C[vt]==7||t.C[vt]==16)for(Q1=0;Q1<t.e[vt];Q1++)Wt=t.i[vt][Q1],t.C[Wt]==7&&t.e[Wt]+n.op(t,Wt)==2&&++BA,t.C[Wt]==8&&t.e[Wt]+n.op(t,Wt)==1&&++R0,t.C[Wt]==16&&t.e[Wt]+n.op(t,Wt)==1&&++C0;St=t.C[vt]==1,t.C[vt]==6&&(it=R0==2,z=n.fm(t,n.Xo(t,A,vt))==2,D1=n.fm(t,n.Xo(t,A,vt))==1&&R0==1),t.C[vt]==7&&(Bt=n.fm(t,n.Xo(t,A,vt))==2,n.fm(t,n.Xo(t,A,vt))==1&&R0==1&&(gt=t.e[vt]+n.op(t,vt)==2||n.vp(t,vt)+n.op(t,vt)==3,It=n.vp(t,vt)+n.op(t,vt)==4),Nt=R0>=2),t.C[vt]==16&&(s1=C0==1,I1=n.fm(t,n.Xo(t,A,vt))==1||n.fm(t,n.Xo(t,A,vt))==2&&R0+BA>1,G1=n.fm(t,n.Xo(t,A,vt))==2&&R0+BA==1),w1=t.C[vt]==15||t.C[vt]==17;}if(D1||gt||St)return 35;if(it||Nt||It||s1||I1||w1)return 32;if(z||Bt||G1)return 7;}break;case 9:if(t.j[A]==1)return 11;if(t.j[A]==0)return 89;break;case 11:if(t.j[A]==0)return 93;break;case 12:if(t.j[A]==0)return 99;break;case 14:return 19;case 15:if(t.e[A]+n.op(t,A)==4)return 25;if(t.e[A]+n.op(t,A)==3)return 26;if(t.e[A]+n.op(t,A)==2)return 75;break;case 16:if(t.e[A]+n.op(t,A)==3||t.e[A]+n.op(t,A)==4){for(eA=0,C0=0,lt=false,V=0;V<t.e[A];V++)vt=t.i[A][V],t.C[vt]==6&&n.fm(t,n.Xo(t,A,vt))==2&&(lt=true),(t.j[vt]==1&&t.C[vt]==8||t.e[vt]+n.op(t,vt)==2&&t.C[vt]==7)&&++eA,t.j[vt]==1&&t.C[vt]==16&&++C0;return t.e[A]+n.op(t,A)==3&&eA==2&&lt||t.e[A]+n.op(t,A)==4?18:eA>0&&C0>0||eA==2&&!lt?73:17;}if(t.e[A]+n.op(t,A)==2){for(c1=false,V=0;V<t.e[A];V++)vt=t.i[A][V],t.C[vt]==8&&n.fm(t,n.Xo(t,A,vt))==2&&(c1=true);return c1?74:15;}if(t.e[A]+n.op(t,A)==1){for(mn=0,lt=false,V=0;V<t.e[A];V++){for(vt=t.i[A][V],Q1=0;Q1<t.e[vt];Q1++)Wt=t.i[vt][Q1],t.C[Wt]==16&&t.e[Wt]+n.op(t,Wt)==1&&++mn;t.C[vt]==6&&n.fm(t,n.Xo(t,A,vt))==2&&(lt=true);}return lt&&mn!=2?16:72;}break;case 17:if(t.j[A]==4){for(nA=0,V=0;V<t.e[A];V++)vt=t.i[A][V],t.C[vt]==8&&++nA;if(nA==4)return 77;}if(t.j[A]==1)return 12;if(t.j[A]==0)return 90;break;case 19:if(t.j[A]==0)return 94;break;case 20:if(t.j[A]==0)return 96;break;case 26:if(t.j[A]==0){if(t.s[A]==2)return 87;if(t.s[A]==3)return 88;}break;case 29:if(t.j[A]==0){if(t.s[A]==1)return 97;if(t.s[A]==2)return 98;}break;case 30:if(t.j[A]==0)return 95;break;case 35:if(t.j[A]==1)return 13;if(t.j[A]==0)return 91;break;case 53:if(t.j[A]==1)return 14;}return 0;},P1.sF=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;for(r=0;r<t.e[A];r++)switch(L=t.i[A][r],t.C[L]){case 6:return 5;case 7:switch(P1.rF(t,L)){case 8:case 39:case 62:case 67:case 68:return 23;case 34:case 54:case 55:case 56:case 58:case 81:return 36;case 9:return 27;default:return 28;}case 8:switch(P1.rF(t,L)){case 49:return 50;case 51:return 52;case 70:return 31;case 6:for(P=false,i=false,D=false,u=false,Q=0;Q<t.e[L];Q++){if(y=t.i[L][Q],t.C[y]==6)for(l=0;l<t.e[y];l++)T=t.i[y][l],T!=L&&(e=n.Xo(t,y,T),(t.C[T]==6||t.C[T]==7)&&(n.fm(t,e)==2||e<t.g&&n.pt(t.p,e))&&(P=true),t.C[T]==8&&n.fm(t,e)==2&&(i=true));t.C[y]==15&&(D=true),t.C[y]==16&&(u=true);}if(i||D)return 24;if(P)return 29;if(u)return 33;default:return 21;}case 14:return 5;case 16:case 15:return 71;}return 0;},P1.tF=function(t,A){return t.a[A]>-1?t.a[A]:t.C[A]==1?P1.sF(t,A):P1.rF(t,A);},P1.uF=function(t,A){var e,r;if((t.u[A]&V0)!=0){for(r=(n.bw(t,7),t.p),e=0;e<r.j.a.length;e++)if(U.xj(r.k,e).length==6&&n.qt(r,e,A))return  true;}return  false;},P1.vF=function(t,A){var e,r,P;for(P=(n.bw(t,7),t.p),r=0,e=0;e<P.j.a.length;e++)n.ct(P,e,A)>=0&&++r;return r;},P1.wF=function(t,A,e){var r,P;if((t.u[A]&V0)==0||(t.u[e]&V0)==0)return  false;for(P=(n.bw(t,7),t.p),r=0;r<P.j.a.length;r++)if(U.xj(P.k,r).length==5&&n.qt(P,r,A)&&n.qt(P,r,e))return  true;return  false;},P1.xF=function(t,A){var e,r;if(t.C[A]==7&&t.e[A]+n.op(t,A)>=3){for(e=0;e<t.e[A];e++)if(r=t.i[A][e],t.C[r]==8&&t.e[r]+n.op(t,r)==1)return  true;}return  false;},P1.yF=function(t,A,e){var r,P;if(A<t.f&&n.ot(t.p,A)){for(P=(n.bw(t,7),t.p),r=0;r<P.j.a.length;r++)if(!(U.xj(P.k,r).length!=e||!n.qt(P,r,A))&&t.b[r]==0)return  true;}return  false;},P1.zF=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;if(v=(n.bw(t,7),t.p),!v.e[A])return 1;if(U.xj(v.k,A).length==6){for(r=U.xj(v.j,A),P=0,i=r.length;P<i;++P)if(e=r[P],n.vp(t,e)+n.op(t,e)!=t.e[e]+n.op(t,e)+1)return 1;for(Q=U.xj(v.k,A),l=0,L=Q.length;l<L;++l)if(u=Q[l],y=j(F(o.Q7,1),H,6,15,[-1,-1]),n.fm(t,u)==1){for(c=0;c<=1;c++)for(D=t.D[c][u],T=0;T<t.e[D];T++)if(b=t.i[D][T],!n.qt(v,A,b)&&n.fm(t,n.Xo(t,D,b))==2){y[c]=b;break;}if(y[0]>-1&&y[1]>-1)for(k=0;k<v.j.a.length;k++){if(n.qt(v,k,y[0])&&n.qt(v,k,y[1])&&t.b[k]==2)return 2;if(n.qt(v,k,y[0])&&n.qt(v,k,y[1])&&t.b[k]!=0)return 1;}}}if(U.xj(v.k,A).length==5)for(g=1,r=U.xj(v.j,A),P=0,i=r.length;P<i;++P){if(e=r[P],n.vp(t,e)+n.op(t,e)==t.e[e]+n.op(t,e)&&g>0){--g;continue;}if(n.vp(t,e)+n.op(t,e)!=t.e[e]+n.op(t,e)+1)return 1;}return 0;},P1.AF=function(t,A,e){return P1.BF(t,A,A.D[0][e],A.D[1][e]);},P1.BF=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c;for(P=A.a[e],i=A.a[r],L=n.Xo(A,e,r),y=true,c=(n.bw(A,7),A.p),T=0;T<c.j.a.length&&y;T++)for(u=U.xj(c.k,T),Q=0,l=u.length;Q<l;++Q)if(D=u[Q],D==L&&A.b[T]==0){y=false;break;}return n.fm(A,L)==1&&y&&(rt.$D(t.b,P)&&rt.$D(t.b,i)||rt.eE(t.b,P)&&rt.eE(t.b,i))?1:0;},P1.CF=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;for(u=P1.DF(A),D=B(o.O7,Ft,6,u.length,15,1),r=0;r<u.length;r++){if(k=A.a[r],c=u[r],v=rt.uE(t.e,k),e=rt._D(t.b,k),b=0,g=0,d.Math.abs(v)<we)for(l=0;l<A.e[r];l++)y=A.i[r][l],u[y]<0&&(c+=u[y]/(2*A.e[y]));if(k==62)for(L=0;L<A.e[r];L++)y=A.i[r][L],u[y]>0&&(c-=u[y]/2);for(Q=0;Q<A.e[r];Q++)y=A.i[r][Q],P=n.Xo(A,r,y),T=A.a[y],i=P1.BF(t,A,A.D[0][P],A.D[1][P]),g+=rt.vE(t.e,i,k,T),b+=u[y];D[r]=(1-e*v)*c+v*b+g;}return D;},P1.DF=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O;D=B(o.O7,Ft,6,t.q,15,1);t:for(i=0;i<D.length;i++){switch(O=t.a[i],D[i]=0,O){case 32:case 72:for(l=0;l<t.e[i];l++){for(v=t.i[i][l],E=t.a[v],g=0,k=0,y=0;y<t.e[v];y++)S=t.i[v][y],t.C[S]==7&&t.e[S]+n.op(t,S)==2&&!(S<t.f&&n.ot(t.p,S))&&++g,(t.C[S]==8||t.C[S]==16)&&t.e[S]+n.op(t,S)==1&&++k;if(t.C[v]==16&&k==2&&g==1&&(g=0),t.C[v]==6&&k>0){D[i]=k==1?-1:-(k-1)/k;continue t;}if(E==45&&k==3){D[i]=-0.3333333333333333;continue t;}if(E==25&&k>0){D[i]=k==1?0:-(k-1)/k;continue t;}if(E==18&&k>0){D[i]=g+k==2?0:-(g+k-2)/k;continue t;}if(E==73&&k>0){D[i]=k==1?0:-(k-1)/k;continue t;}if(E==77&&k>0){D[i]=-1/k;continue t;}}break;case 76:for(G=0,K=(n.bw(t,7),t.p),V=0;V<K.j.a.length;V++)if(n.qt(K,V,i)){G=V;break;}if(G<K.j.a.length){for(b=0,e=U.xj(K.j,G),r=0,P=e.length;r<P;++r)A=e[r],A>-1&&t.a[A]==76&&++b;if(b>0){D[i]=-1/b;continue t;}}break;case 55:case 56:case 81:for(D[i]=t.s[i],c=1,q=0,u=B(o.vfb,Tt,6,t.q,16,1),u[i]=true;c>q;)for(q=c,L=0;L<t.q;L++)if(u[L]){for(y=0;y<t.e[L];y++)if(v=t.i[L][y],E=t.a[v],!(E!=57&&E!=80))for(T=0;T<t.e[v];T++)S=t.i[v][T],p=t.a[S],!(p!=55&&p!=56&&p!=81)&&(u[S]||(u[S]=true,D[i]+=t.s[S],++c));}D[i]/=c;continue t;case 61:for(Q=0;Q<t.e[i];Q++)if(v=t.i[i][Q],t.a[v]==42){D[i]=1;continue t;}continue t;case 34:case 49:case 51:case 54:case 58:case 92:case 93:case 94:case 97:D[i]=1;continue t;case 87:case 95:case 96:case 98:case 99:D[i]=2;continue t;case 88:D[i]=3;continue t;case 35:case 62:case 89:case 90:case 91:D[i]=-1;continue t;}D[i]=0;}return D;},P1.EF=function(t,A,e,r,P,i){var D,u,Q,l,L,y;return l=n.Xo(A,r,P),D=P1.BF(t,A,e,r),u=P1.BF(t,A,r,P),Q=P1.BF(t,A,P,i),y=u,u==0&&n.fm(A,l)==1&&(D==1||Q==1)&&(y=2),L=P1.FF(A,e,r,P,i),L==4&&n.Xo(A,e,P)==-1&&n.Xo(A,r,i)==-1?40+y:L==5&&(A.a[e]==1||A.a[r]==1||A.a[P]==1||A.a[i]==1)?50+y:y;},P1.FF=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T;if(n.Xo(t,A,e)==-1||n.Xo(t,e,r)==-1||n.Xo(t,r,P)==-1)return 0;if(n.Xo(t,P,A)>=0)return 4;for(y=(n.bw(t,7),t.p),T=new U.rub(),U.qub(T,h.qjb(A)),U.qub(T,h.qjb(e)),U.qub(T,h.qjb(r)),U.qub(T,h.qjb(P)),l=0;l<y.j.a.length;l++)if(U.xj(y.k,l).length==5){for(L=new U.rub(),D=U.xj(y.j,l),u=0,Q=D.length;u<Q;++u)i=D[u],U.qub(L,h.qjb(i));if(U.Ji(L,T))return 5;}return 0;},x(426,1,{}),o.V9=N(426),n1.HF=function(){n1.HF=f1,n1.GF=(h.Pkb(),d1);},n1.IF=function(t){},n1.JF=function(t){var A,e,r;for(A=0,e=0,r=0;r<t.g.a.length+t.f.a.length;r++)A+=n.Ul(r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length))*(r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length)).r,e+=(r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length)).r;return e!=0?e/A:1;},n1.KF=function(t,A){n1.HF();var e,r,P,i;n1.IF(this),r=new At.AH(t);try{for(i=new Rt.thb(),Rt.shb(i,"$RXN"+n1.GF),h.Bkb(i.a,A??""),Rt.shb(i,n1.GF+(""+n1.GF)),Rt.shb(i,Tr+At.VH(t,!0,27)),Rt.shb(i,n1.GF),Rt.shb(i,"  "+r.g.a.length+"  "+r.f.a.length+n1.GF),P=n1.JF(r),e=0;e<r.g.a.length+r.f.a.length;e++)Rt.shb(i,"$MOL"+n1.GF),n.$r(new n.bs(e<r.g.a.length?U.xj(r.g,e):U.xj(r.f,e-r.g.a.length),P,null),i);this.a=i.a;}catch(D){if(D=C1(D),Zt(D,19))h.Pkb();else throw st(D);}},x(279,1,{},n1.KF),s.a=null,o.W9=N(279),n1.LF=function(t){var A,e;return A=new At.zH(),e=new Rt.ehb(new Rt.phb(t)),n1.MF(A,e),A;},n1.MF=function(t,A){var e,r;if(r=Rt.dhb(A),e=false,r==null||!h.akb(w.SAb(r).substr(0,4),"$RXN"))throw st(new h.cC("'$RXN' tag not found"));return h.akb(r,c9)?e=n1.OF(t,A,false):e=n1.NF(t,A,false),e;},n1.NF=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;if(l=h.skb(Rt.dhb(A)),w.SAb(l).length==0||(t.e=l),Rt.dhb(A),r=Rt.dhb(A),!e&&h.akb(w.SAb(r).substr(0,13),Tr)&&(P=(w.NAb(13,w.SAb(r).length+1),w.SAb(r).substr(13)),At.UH(P,true,t)))return  true;for(b=Rt.dhb(A),c=h.Mib(h.skb((w.MAb(0,3,w.SAb(b).length),w.SAb(b).substr(0,3))),r1,qt),y=h.Mib(h.skb((w.MAb(3,6,w.SAb(b).length),w.SAb(b).substr(3,3))),r1,qt),u=new n.Cs(),D=0;D<c;D++){if(b=Rt.dhb(A),b==null||!h.akb(w.SAb(b).substr(0,4),"$MOL"))throw st(new h.cC(b9));T=new n.tw(),Q=new h.Dkb();do b=Rt.dhb(A),Q.a+=""+b,Q.a+=d1;while(b!=null&&!h.akb(w.SAb(b).substr(0,6),FA));if(b==null)throw st(new h.cC(w9));n.xs(u,T,new Rt.ehb(new Rt.phb(Q.a))),U.sj(t.g,T),t.d=-1;}for(i=0;i<y;i++){if(b=Rt.dhb(A),b==null||!h.akb(w.SAb(b).substr(0,4),"$MOL"))throw st(new h.cC(b9));L=new n.tw(),Q=new h.Dkb();do b=Rt.dhb(A),Q.a+=""+b,Q.a+=d1;while(b!=null&&!h.akb(w.SAb(b).substr(0,6),FA));if(b==null)throw st(new h.cC(w9));n.xs(u,L,new Rt.ehb(new Rt.phb(Q.a))),U.sj(t.f,L),t.d=-1;}return  true;},n1.OF=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;if(l=h.skb(Rt.dhb(A)),w.SAb(l).length==0||(t.e=l),Rt.dhb(A),r=Rt.dhb(A),!e&&h.akb(w.SAb(r).substr(0,13),Tr)&&(P=(w.NAb(13,w.SAb(r).length+1),w.SAb(r).substr(13)),At.UH(P,true,t)))return  true;if(b=Rt.dhb(A),D=new n.Cs(),b!=null&&h.akb(w.SAb(b).substr(0,13),"M  V30 COUNTS")){if(c=h.skb((w.NAb(13,w.SAb(b).length+1),w.SAb(b).substr(13))),L=h.okb(c," "),T=h.Mib(L[0],r1,qt),y=h.Mib(L[1],r1,qt),T>0){if(b=Rt.dhb(A),h.akb(g9,b))for(i=0;i<T;i++){u=new n.tw(),Q=new h.Dkb(),Q.a+=B9;do b=Rt.dhb(A),Q.a+=""+b,Q.a+=d1;while(b!=null&&!h.akb(w.SAb(b).substr(0,15),LP));n.xs(D,u,new Rt.ehb(new Rt.phb(Q.a))),U.sj(t.g,u),t.d=-1;}b=Rt.dhb(A);}if(y>0&&(b=Rt.dhb(A),h.akb(k9,b))){for(i=0;i<y;i++){u=new n.tw(),Q=new h.Dkb(),Q.a+=B9;do b=Rt.dhb(A),Q.a+=""+b,Q.a+=d1;while(b!=null&&!h.akb(w.SAb(b).substr(0,15),LP));n.xs(D,u,new Rt.ehb(new Rt.phb(Q.a))),U.sj(t.f,u),t.d=-1;}b=Rt.dhb(A);}return  true;}return  false;},n1.QF=function(){n1.QF=f1,n1.PF=(h.Pkb(),d1);},n1.RF=function(t){},n1.SF=function(t){var A,e,r;for(A=0,e=0,r=0;r<t.g.a.length+t.f.a.length;r++)A+=n.Ul(r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length))*(r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length)).r,e+=(r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length)).r;return e!=0?e/A:1;},n1.TF=function(t,A){n1.QF();var e,r,P,i,D,u;n1.RF(this),i=new At.AH(t);try{if(u=new Rt.thb(),Rt.shb(u,c9+n1.PF),h.Bkb(u.a,A??""),Rt.shb(u,n1.PF+(""+n1.PF)),Rt.shb(u,Tr+At.VH(t,!0,11)),Rt.shb(u,n1.PF),P=i.g.a.length,r=i.f.a.length,Rt.shb(u,Z8+P+" "+r+n1.PF),D=n1.SF(i),P>0){for(Rt.shb(u,g9+n1.PF),e=0;e<i.g.a.length;e++)Rt.shb(u,n.Os(U.xj(i.g,e),D));Rt.shb(u,"M  V30 END REACTANT"+n1.PF);}if(r>0){for(Rt.shb(u,k9+n1.PF),e=0;e<i.f.a.length;e++)Rt.shb(u,n.Os(U.xj(i.f,e),D));Rt.shb(u,"M  V30 END PRODUCT"+n1.PF);}Rt.shb(u,FA+n1.PF),this.a=u.a;}catch(Q){if(Q=C1(Q),Zt(Q,19))h.Pkb();else throw st(Q);}},x(280,1,{},n1.TF),s.a=null,o.X9=N(280),n1.VF=function(){n1.VF=f1,n1.UF=j(F(o.Ocb,1),yt,2,6,["Idorsia No","Actelion No","ID","IDNUMBER","COMPOUND_ID","NAME","COMPND"]);},n1.WF=function(t){var A,e,r,P,i,D,u,Q,l;if(!t.i)return  false;h.whb(t.g,0),h.whb(t.b,0),t.f=null,l=false,e=-1,r=null,t.c=t.d==null?null:B(o.Ocb,yt,2,t.d.length,6,1),t.e=-1;do{try{if(Q=Rt.dhb(t.i),Q==null)return h.whb(t.g,0),!1;}catch(L){if(L=C1(L),Zt(L,75))return h.whb(t.g,0),false;throw st(L);}if(l)h.Ikb(t.b,Q),h.Ekb(t.b,10);else if(h.akb(w.SAb(Q).substr(0,1),">"))l=true,h.Ikb(t.g,FA),h.Ekb(t.g,10),h.Ikb(t.b,Q),h.Ekb(t.b,10);else {h.Ikb(t.g,Q),h.Ekb(t.g,10),h.akb(w.SAb(Q).substr(0,6),FA)&&(l=true);continue;}if(t.d!=null)if(w.SAb(Q).length==0)e=-1;else if(e==-1){if(r=n1.YF(Q),r!=null){for(e=-1,A=0;A<t.d.length;A++)if(h.akb(r,t.d[A])){e=A;break;}if(t.e==-1){for(i=n1.UF,D=0,u=i.length;D<u;++D)if(P=i[D],h.akb(r,P)){t.e=e;break;}}}}else t.c[e]==null?t.c[e]=Q:t.c[e]=h.Yjb(h.Yjb(t.c[e],d1),Q);}while(!h.akb(w.SAb(Q).substr(0,4),le));return  true;},n1.XF=function(t,A){var e,r,P,i;for(i=0,r=new n.rx();i<A;){try{P=Rt.dhb(t.i);}catch(D){if(D=C1(D),Zt(D,75))break;throw st(D);}if(P==null)break;h.akb(w.SAb(P).substr(0,4),le)&&++i,h.akb(w.SAb(P).substr(0,1),">")&&(e=n1.YF(P),e!=null&&n.qx(r,e));}t.d=U.Lj(r.b,B(o.Ocb,yt,2,0,6,1));},n1.YF=function(t){var A,e,r,P;if(w.SAb(t).length==0||(w.NAb(0,w.SAb(t).length),w.SAb(t).charCodeAt(0)!=62))return null;for(r=1,P=0,A=0;r<w.SAb(t).length;){if(w.NAb(r,w.SAb(t).length),w.SAb(t).charCodeAt(r)==60){if(P!=0)return null;P=r;}else if(w.NAb(r,w.SAb(t).length),w.SAb(t).charCodeAt(r)==62){if(A!=0)return null;A=r;}++r;}if(P!=0&&P<A)return w.MAb(P+1,A,w.SAb(t).length),w.SAb(t).substr(P+1,A-(P+1));if(r=w.SAb(t).indexOf("DT",1),r==-1)return null;for(e=r+2;w.SAb(t).length>e&&AA((w.NAb(e,w.SAb(t).length),w.SAb(t).charCodeAt(e)));)++e;return e==r+2?null:(w.MAb(r,e,w.SAb(t).length),w.SAb(t).substr(r,e-r));},n1.ZF=function(t,A){return t.c==null?null:t.c[A];},n1.$F=function(t){return t.d==null&&n1.XF(t,10240),t.d;},n1._F=function(t,A){return t.d==null&&n1.XF(t,A),t.d;},n1.aG=function(t){var A,e;return t.f||(e=new n.Cs(),t.a&&(e.a=true),t.f=n.fs(e,(A=t.g.a,A)),t.f&&(t.f.P==null||w.SAb(t.f.P).length==0)&&n.Un(t.f,t.e!=-1&&t.c!=null?t.c[t.e]:t.f?t.f.P:null)),t.f;},n1.bG=function(t,A){n1.VF(),this.d=A,this.i=new Rt.ehb(t),this.g=new h.Lkb(),this.b=new h.Lkb();},x(281,426,{},n1.bG),s.a=false,s.e=0,o.Y9=N(281),et.fG=function(){et.fG=f1,et.cG=j(F(o.R7,1),e0,6,14,[524290,524292,589826,589827,589828,589829,589832,589833,589837,589838,598019,598020,598024,606211,688130,688131,688132,688137,688142,696324,819202,819203,819204,819205,819214,827396,1048580,1114114,1114115,1114116,1114120,1212420,1220611,1343492,1343496,1638402,1638403,17367042,17367043,17367044,17367045,17367048,17367049,17367053,17367054,17375236,17465346,17465348,17465353,17473540,1208483842,1208483848,1208483912,1208549378,1208549379,1208549380,1208549384,1208549388,1208549442,1208549443,1208549444,1208549448,1208557571,1208565763,1208565827,1409875970,1409875971,1409875972,1409875976,1409876034,1409876035,1409876036,1409876040,1409884163,1409974274,1409974280,1409974338,1678245892,1678311426,1678311427,1678311428,1678311432,1678311490,1678311491,1678311492,1678311496,1678409730,1678409731,1678409732,1678409794,1678540802,1678540808,1678540810,1678540866,2282291202,2282291203,2282291266,2282291267,2282389506,2282389507,2282389570,2282389571,2282389572,2282520578,2282520579,2282520642,2282520643,2282537027,2483617794,2483617795,2483617858,2483617859,2483617860,2483617864,2483716098,2483716099,2483716162,2483716163,2484150275,2484248579,2752675842,3356033026,3356139523,3557359618,3557457922,3557588994,34510798914,34510798915,34510798916,34510798920,34510807107,34510815299,34712125506,34712125507,34712125508,34712125512,34712223810,34712223811,34712223812,34712223816,34980560962,34980560963,34980659266,34980659267,34980659268,35568287746,35568287747,35568287748,35568287752,35568287756,35568287810,35568287811,35568287812,35568287816,35568287820,35568386050,35568386051,35568386052,35568386114,35568386120,35568517122,35568517123,35568517124,35568517128,35568517186,35568517187,35568517188,35568517192,35568812034,35568812035,35568812098,35568812099,35568910338,35568910339,35568910403,35569336322,35569434626,35585064962,35585064963,35585064964,35585064968,35585065026,35585065027,35585065028,35585065032,35769614338,35769614340,35769614402,35769712642,35770138626,35770236930,35786391554,35786391555,2475109646403,d9,m9,2475109711938,2475109711939,2475109720067,2475109720131,2475109728259,2475109728323,v9,F9,2887426572354,2887426572355,j9,p9,2887627898946,M9,2887627997250,R9,C9,3437182386242,3437182386243,S9,3437383712771,3437383712834,H9,E9,q9,V9,4674132967490,4674132967491,4674132975619,I9,4674334294082,4674334294083,4674334302211,4674334302275,O9,4674334392386,x9,4674602729538,4674602827842,N9,4674602958914,G9,z9,5086449827906,5086449827907,5086449827912,J9,5086651154498,5086651162627,K9,Z9,5086651252802,5086651252808,5086651260931,Y9,5086919589954,X9,5086919688258,5086919819330,5636205641794,_9,5636406968386,W9,5636407066690,5636675403842,$9,5636675502146,{l:589890,m:73760,h:4},{l:589891,m:73760,h:4},{l:598083,m:73760,h:4},{l:688194,m:73760,h:4},{l:688195,m:73760,h:4},{l:696387,m:73760,h:4},{l:819266,m:73760,h:4},{l:65602,m:73764,h:4},{l:65603,m:73764,h:4},{l:589890,m:172064,h:4},{l:589891,m:172064,h:4},{l:598083,m:172064,h:4},{l:688194,m:172064,h:4},{l:688195,m:172064,h:4},{l:696387,m:172064,h:4},{l:819266,m:172064,h:4},{l:65602,m:172068,h:4},{l:65603,m:172068,h:4},{l:589890,m:172112,h:4},{l:589891,m:172112,h:4},{l:688194,m:172112,h:4},{l:819266,m:172112,h:4},{l:65602,m:172116,h:4},{l:65608,m:172116,h:4},{l:163906,m:172116,h:4},{l:589890,m:303136,h:4},{l:688194,m:303136,h:4},{l:819266,m:303136,h:4},{l:65602,m:303140,h:4},{l:589890,m:303184,h:4},{l:688194,m:303184,h:4},{l:819266,m:303184,h:4},{l:65602,m:303188,h:4},{l:65602,m:303252,h:4},a9,tf,{l:589890,m:590112,h:4},{l:589891,m:590112,h:4},Af,nf,{l:589890,m:590160,h:4},{l:589891,m:590160,h:4},ef,{l:688194,m:590160,h:4},rf,Pf,{l:589890,m:590224,h:4},{l:589891,m:590224,h:4},{l:688131,m:590224,h:4},{l:688194,m:590224,h:4},ff,Df,{l:589890,m:590368,h:4},uf,{l:688194,m:590368,h:4},{l:696323,m:590368,h:4},Qf,of,{l:819266,m:590368,h:4},lf,Lf,{l:589890,m:590416,h:4},{l:589891,m:590416,h:4},{l:589896,m:590416,h:4},sf,{l:688136,m:590416,h:4},{l:688194,m:590416,h:4},{l:696323,m:590416,h:4},Uf,{l:819266,m:590416,h:4},{l:688130,m:590480,h:4},{l:688194,m:590480,h:4},{l:65602,m:598052,h:4},{l:65603,m:598052,h:4},{l:65602,m:598100,h:4},{l:65603,m:598100,h:4},{l:73795,m:598100,h:4},{l:163906,m:598100,h:4},{l:163907,m:598100,h:4},{l:65602,m:598164,h:4},{l:163906,m:598164,h:4},yf,Tf,{l:589890,m:598304,h:4},{l:589891,m:598304,h:4},hf,{l:688194,m:598304,h:4},cf,{l:1114178,m:598304,h:4},bf,{l:1212424,m:598304,h:4},{l:1212482,m:598304,h:4},wf,gf,{l:589890,m:688416,h:4},Bf,{l:589890,m:688464,h:4},{l:589890,m:688528,h:4},kf,df,{l:688130,m:688784,h:4},{l:65602,m:696356,h:4},{l:65602,m:696404,h:4},{l:163906,m:696404,h:4},{l:589826,m:696608,h:4},{l:1114178,m:696608,h:4},{l:1212418,m:696608,h:4},{l:589826,m:696612,h:4},{l:524355,m:590112,h:288},{l:589825,m:590112,h:288},mf,{l:589890,m:590112,h:288},{l:598019,m:590112,h:288},{l:598083,m:590112,h:288},vf,{l:589890,m:590112,h:336},{l:598019,m:590112,h:336},{l:598083,m:590112,h:336},Ff,{l:589890,m:688416,h:336},jf,{l:589890,m:688464,h:336},pf,Mf,{l:589827,m:590112,h:400},{l:589890,m:590112,h:400},{l:589891,m:590112,h:400},Rf,Cf,Sf,Hf,{l:589890,m:819488,h:400},Ef,qf,Vf,If,{l:589827,m:819488,h:544},Of,xf,Nf,{l:688199,m:688464,h:592},Gf,zf,Jf,{l:688202,m:819600,h:592},Kf,{l:589896,m:1212704,h:592},Zf,Yf,{l:688200,m:1212752,h:592},Xf,{l:688135,m:688464,h:656},{l:688135,m:819536,h:656},{l:589891,m:65936,h:8228},_f,{l:589890,m:590112,h:8480},{l:598019,m:590112,h:8480},Wf,{l:589890,m:688416,h:8480},$f,{l:589890,m:688464,h:8480},af,tD,AD,nD,{l:589896,m:1213008,h:8480},eD,{l:688200,m:1213008,h:8480},{l:688135,m:1343824,h:8480},rD,{l:589890,m:590112,h:8484},PD,{l:589890,m:590160,h:8484},{l:1212424,m:590416,h:8484},{l:1212488,m:590416,h:8484},iD,{l:688194,m:598304,h:8484},{l:589826,m:590112,h:8528},{l:589890,m:590112,h:8528}]),et.dG=j(F(o.P7,1),w0,6,15,[.6966999769210815,0,.4885999858379364,-0.47269999980926514,-0.07490000128746033,fD,.273499995470047,.5699999928474426,.7009999752044678,.9534000158309937,-3.6435000896453857,-2.150899887084961,.4975000023841858,-2.19950008392334,-0.2809000015258789,-0.8259999752044678,-0.1784999966621399,-1.620300054550171,-1.0959999561309814,-0.3619999885559082,.13950000703334808,-0.29750001430511475,-1.2907999753952026,1.0161999464035034,DD,-2.5383999347686768,hr,.4291999936103821,-0.5824000239372253,-0.1834000051021576,.1306000053882599,-0.5015000104904175,fD,-0.5257999897003174,.4244000017642975,-0.16099999845027924,-0.2777999937534332,.5110999941825867,-0.435699999332428,-0.10409999638795853,.3424000144004822,-0.061500001698732376,.6035000085830688,.7226999998092651,.43459999561309814,-1.6821000576019287,-0.3310000002384186,-0.49799999594688416,uD,-0.3650999963283539,.45969998836517334,.3384000062942505,.6632999777793884,.4544000029563904,.15970000624656677,.633899986743927,.35040000081062317,.04490000009536743,.34200000762939453,.26109999418258667,.40459999442100525,.5218999981880188,-1.065000057220459,-1.2314000129699707,-1.802299976348877,-0.36320000886917114,-0.4108000099658966,.30570000410079956,-0.14560000598430634,-0.27129998803138733,-0.5192999839782715,.45260000228881836,.5539000034332275,.8374000191688538,-0.7070000171661377,-0.48809999227523804,-0.4099999964237213,0,.14790000021457672,.3447999954223633,.42980000376701355,.5579000115394592,-0.1264999955892563,-0.042500000447034836,.07670000195503235,.6635000109672546,-0.38119998574256897,-0.8367999792098999,1.0286999940872192,-0.10209999978542328,.3587000072002411,-0.5945000052452087,.16920000314712524,-0.121799997985363,.32829999923706055,.22390000522136688,.20430000126361847,.05900000035762787,-0.48350000381469727,.6165000200271606,-0.4011000096797943,.5577999949455261,-0.21639999747276306,-0.017500000074505806,.29809999465942383,.10999999940395355,.27149999141693115,-0.2994999885559082,-0.46700000762939453,.1565999984741211,.046799998730421066,-0.13210000097751617,1.3686000108718872,0,-0.4115999937057495,1.0185999870300293,-0.3935000002384186,.5223000049591064,.3684999942779541,.25769999623298645,1.5192999839782715,.2705000042915344,.3790999948978424,.012000000104308128,-0.33970001339912415,.14830000698566437,.2766000032424927,.35929998755455017,.7714999914169312,.3149999976158142,-1.618499994277954,.18889999389648438,-0.2651999890804291,-0.09650000184774399,.420199990272522,.18709999322891235,-0.3684000074863434,-0.07779999822378159,.8942999839782715,.3693999946117401,.28790000081062317,.4489000141620636,-0.26010000705718994,.4771000146865845,.1923000067472458,.43810001015663147,.16949999332427979,.45249998569488525,.3352000117301941,.1582999974489212,.4036000072956085,-0.04800000041723251,.5023000240325928,-0.26489999890327454,.76910001039505,-0.35519999265670776,1.0300999879837036,-0.11410000175237656,-0.5932000279426575,.17489999532699585,.13130000233650208,-0.18039999902248383,.399399995803833,.22910000383853912,.31690001487731934,.35989999771118164,-0.0038999998942017555,-0.2955999970436096,.4408999979496002,-0.16089999675750732,.3774999976158142,-0.13459999859333038,.2838999927043915,.5128999948501587,.1265999972820282,.4293999969959259,.28060001134872437,.49070000648498535,QD,.219200000166893,.15649999678134918,.6934999823570251,.3617999851703644,.6735000014305115,.5777999758720398,-0.5636000037193298,.5569000244140625,.30379998683929443,-0.32760000228881836,-0.6991999745368958,.010300000198185444,-0.4659000039100647,oD,hr,.18410000205039978,.707099974155426,.12269999831914902,.7949000000953674,-0.6592000126838684,-1.3148000240325928,-0.4066999852657318,-0.1316000074148178,-0.4925000071525574,-0.09290000051259995,-0.4352000057697296,-0.2206999957561493,-0.9959999918937683,-0.723800003528595,-0.5468999743461609,-1.2939000129699707,-0.01360000018030405,.065700002014637,.7189000248908997,.05700000002980232,.661899983882904,-0.6381000280380249,-0.8072999715805054,.23549999296665192,.30480000376701355,-0.019899999722838402,-0.07519999891519547,.44609999656677246,.1559000015258789,1.1167999505996704,-1.8039000034332275,.23649999499320984,-0.22059999406337738,.4480000138282776,-1.1339999437332153,-0.5652999877929688,-0.40529999136924744,-0.13609999418258667,.2198999971151352,.053599998354911804,-0.020999999716877937,.6984999775886536,.9642999768257141,-0.41519999504089355,-1.0369000434875488,-0.18299999833106995,.5882999897003174,-0.29179999232292175,-0.5293999910354614,-0.6541000008583069,.9473000168800354,-0.19059999287128448,-0.8483999967575073,-0.3456999957561493,.9541000127792358,1.4230999946594238,-0.7924000024795532,lD,LD,-0.2596000134944916,.8381999731063843,-0.4415999948978424,-0.37040001153945923,-0.7487000226974487,-0.10790000110864639,-0.29919999837875366,-0.3276999890804291,.025100000202655792,-0.9187999963760376,.10939999669790268,.8230999708175659,-3.233299970626831,.03500000014901161,oD,sD,.2791000008583069,.3206000030040741,.5662000179290771,-0.3783999979496002,.4032000005245209,-1.794800043106079,-0.15539999306201935,.3785000145435333,.05339999869465828,-0.16529999673366547,-0.09870000183582306,-0.10050000250339508,-0.6460999846458435,.8034999966621399,-0.24050000309944153,-0.12380000203847885,-0.35760000348091125,.09610000252723694,-0.6401000022888184,.2029000073671341,.23589999973773956,.4950999915599823,.19210000336170197,-0.37450000643730164,.34630000591278076,.289900004863739,-0.15330000221729279,-0.4169999957084656,.37700000405311584,.6998000144958496,.593999981880188,.5911999940872192,-0.5570999979972839,.023800000548362732,-0.2475000023841858,.030700000002980232,-0.38749998807907104,-0.7437000274658203,.5144000053405762,.00570000009611249,.765500009059906,.1720000058412552,-2.5624001026153564,-0.30660000443458557,.36469998955726624,.17270000278949738,-0.03290000185370445,-0.18930000066757202,.07020000368356705,-1.2453999519348145,.14959999918937683,DD,.4146000146865845,-0.2667999863624573,-0.11060000211000443,.03620000183582306,-0.3188999891281128,-0.7278000116348267,-0.08940000087022781,-0.22769999504089355,-0.2393999993801117,1.011199951171875,-0.2962000072002411,.7775999903678894,.2944999933242798,-0.22339999675750732,.27639999985694885,.8011000156402588,-0.17440000176429749,.15809999406337738,-1.7552000284194946,-0.384799987077713,.5993000268936157,.5267999768257141,-0.04170000180602074,.4733000099658966,-0.3400999903678894,-0.14499999582767487,.7088000178337097,-0.13179999589920044,.04259999841451645,-0.5027999877929688,.3831999897956848,-0.011800000444054604,-0.4357999861240387,.3749000132083893,-0.12030000239610672,-0.5648000240325928,-0.1972000002861023,-0.8769000172615051,-0.3675000071525574,-0.2003999948501587,-0.6069999933242798,-0.18569999933242798,.3467999994754791,-0.36239999532699585,.5357999801635742,-0.3700999915599823,.13359999656677246,.9545000195503235,.11389999836683273,-0.16990000009536743,hr,.2890999913215637,.2612999975681305,-0.03440000116825104,-1.9498000144958496,-2.024899959564209,-0.6004999876022339,-0.6258000135421753,-1.2348999977111816,.328000009059906,-0.54339998960495,-0.7712000012397766,-0.9057000279426575,-0.16680000722408295,-0.9904999732971191,-0.03720000013709068,-1.1638000011444092,.12620000541210175,-0.5248000025749207,-0.15379999577999115,-0.36820000410079956,.3249000012874603,.06499999761581421,.051100000739097595,-0.46070000529289246,.22310000658035278,.28220000863075256,.1396999955177307,-0.49380001425743103,.39480000734329224,-0.4074999988079071,-0.6410999894142151,-0.009100000374019146,-0.13330000638961792,-0.5192000269889832,-0.16609999537467957,hr,-0.07069999724626541,.4805999994277954,.38280001282691956,.22290000319480896,.6159999966621399,.33709999918937683,.188400000333786,.13809999823570251,uD,.2833000123500824,-0.1225999966263771,-3.9189000129699707,-0.459199994802475,-0.3434999883174896,-0.6654000282287598,-0.5055999755859375,-0.863099992275238,.15360000729560852,-0.6427000164985657,-0.08839999884366989,-0.0471000000834465,.11060000211000443,.382099986076355,-0.23919999599456787,-0.4050999879837036,.08910000324249268,-0.6972000002861023,-0.4699000120162964,.09220000356435776,.08060000091791153,-0.6773999929428101,-0.062199998646974564,-0.9300000071525574,.13369999825954437]);},et.gG=function(t){var A,e,r,P,i;for(e=0,n.lq(t),n.bw(t,7),A=0;A<t.f;A++)try{i=n.ff(t,A,6241),P=(r=Lt.N4(et.eG,h.Fjb(i)),r<0?-1:r),P!=-1&&(e+=et.dG[P]);}catch(D){if(D=C1(D),!Zt(D,19))throw st(D);}return e;},et.hG=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g;if(P=new et.wG(),U.sj(P.a,new et.tG("cLogP Values are estimated applying an atom-type based increment system.",2)),U.sj(P.a,new et.tG(UD,2)),U.sj(P.a,new et.tG(VP,2)),t){for(n.lq(t),n.bw(t,7),u=0,r=new U.Xwb(),Q=new o0.Hnb("#0.000"),A=0;A<t.f;A++)try{e=n.ff(t,A,6241),g=U.eqb(r,h.Fjb(e)),g?U.Pwb(r,h.Fjb(e),h.qjb(g.a+1)):U.Pwb(r,h.Fjb(e),h.qjb(1));}catch(k){if(k=C1(k),Zt(k,19))++u;else throw st(k);}for(u!=0&&U.sj(P.a,new et.tG("Warning: "+u+" atom type(s) could not be determined.",2)),b=(D=new U.fxb(new U.lxb(new U.sqb(r).a).b),new U.Bqb(D));U.apb(b.a.a);)c=(i=U.dxb(b.a),i.hd()),L=Lt.N4(et.eG,c),(L<0?-1:L)!=-1?et.vG(P,(T=c,U.Xnb(U.Kwb(r,T))+" * "+o0.Enb(Q,et.dG[l=Lt.N4(et.eG,c),l<0?-1:l])+" AtomType: 0x"+h.Djb(c.a)),2):et.vG(P,"Warning: For atom type 0x"+h.Djb(c.a)+" ("+(y=c,U.Xnb(U.Kwb(r,y)))+" times found) is no increment available.",2);}return P;},et.iG=function(){et.fG();var t,A,e,r;if(!et.eG&&!et.eG)for(et.eG=new Lt.Q4(),A=et.cG,e=0,r=A.length;e<r;++e)t=h.Fjb(A[e]),Lt.K4(et.eG,t);},x(213,1,{},et.iG),o.Z9=N(213),et.jG=function(t,A,e,r,P){var i,D,u,Q,l,L;for(i=1/(1+d.Math.exp(t-5)),L=1-1/(1+d.Math.exp(A+5)),l=1/(1+d.Math.exp(.012*e-6)),D=1-1/(1+d.Math.exp(r)),u=(.5+i/2)*(.5+L/2)*(.5+l/2)*(.5+D/2),Q=0;P!=null&&Q<P.length;Q++)P[Q]==2?u*=.8:P[Q]==3&&(u*=.6);return u;},et.mG=function(t,A){var e,r,P,i,D,u,Q,l,L;if(e=new et.wG(),!et.lG)return U.sj(e.a,new et.tG(yD,2)),-999;for(U.sj(e.a,new et.tG("Found sub-structure fragments and their contributions:",2)),U.sj(e.a,new et.tG("(yellow atoms carry at least one more substituent)",2)),l=0,Q=0,i=0,L=new n.ju(1),P=new n.tw(),D=0;D<et.kG.a.a.length;D++)n.Tq(new n._q(false),P,et.pG(et.kG,D)),n.cu(L,A),n._t(L,P),n.Pt(L,1,L.c)>0&&(u=et.qG(et.kG,D),u<-1?l+=u:(Q+=u,++i),et.vG(e,et.pG(et.kG,D),1),U.sj(e.a,new et.tG(""+u,3)));return i==0?-1:(r=l+Q/d.Math.sqrt(i),r=r+.0625*(i-40),t.a=e,r);},et.nG=function(t){var A,e,r,P,i,D;if(!et.lG)return yD;for(i=0,r=0,D=new n.ju(1),e=new n.tw(),P=0;P<et.kG.a.a.length;P++)n.Tq(new n._q(false),e,et.pG(et.kG,P)),n.cu(D,t),n._t(D,e),n.Pt(D,1,D.c)>0&&(i+=et.qG(et.kG,P),++r);return A=r==0?-1:i/d.Math.sqrt(r),A+"	"+r+"	"+t.f;},et.oG=function(){if(!et.lG)try{et.kG=new et.rG(),et.lG=!0;}catch(t){if(t=C1(t),Zt(t,19))h.Pkb();else throw st(t);}},x(273,1,{},et.oG),et.lG=false,o.$9=N(273),et.pG=function(t,A){return U.xj(t.a,A).a;},et.qG=function(t,A){return U.xj(t.a,A).b;},et.rG=function(){var t,A,e,r,P,i;for(i=new Rt.ehb(new Rt.jhb((e=_t.cBb("/resources/druglikenessNoIndex.txt"),q1.Cnb(),e))),this.a=new U.Mj();P=Rt.dhb(i),P!=null;){if(r=h.hkb(P,X1(9)),r==-1)throw st(new h.cC("line without TAB"));t=(w.MAb(0,r,w.SAb(P).length),w.SAb(P).substr(0,r)),A=h.Rib(w.SAb(h.Lib((w.NAb(r+1,w.SAb(P).length+1),w.SAb(P).substr(r+1))))),U.sj(this.a,new et.sG(t,A));}},x(274,1,{},et.rG),o.aab=N(274),et.sG=function(t,A){this.a=t,this.b=A;},x(236,1,{236:1},et.sG),s.b=0,o._9=N(236),et.tG=function(t,A){this.a=t,this.b=A;},x(45,1,{45:1},et.tG),s.b=0,o.cab=N(45),et.uG=function(t){t.a=new U.Mj();},et.vG=function(t,A,e){U.sj(t.a,new et.tG(A,e));},et.wG=function(){et.uG(this);},x(132,1,{},et.wG),o.bab=N(132),et.zG=function(){et.zG=f1,et.xG=j(F(o.Ocb,1),yt,2,6,["[N](-*)(-*)-*","[N](-*)=*","[N]#*","[N](-*)(=*)=* as in nitro","[N](=*)#* middle atom of azide","[N]1(-*)-*-*-1 3-membered ring","[NH](-*)-*","[NH]1-*-*-1 3-membered ring","[NH]=*","[NH2]-*","[N+](-*)(-*)(-*)-*","[N+](-*)(-*)=*","[N+](-*)#* N in isocyano","[NH+](-*)(-*)-*","[NH+](-*)=*","[NH2+](-*)-*","[NH2+]=*","[NH3+]-*","[n](:*):*","[n](:*)(:*):*","[n](-*)(:*):*","[n](=*)(:*):* as in pyridine-N-oxid","[nH](:*):*","[n+](:*)(:*):*","[n+](-*)(:*):*","[nH+](:*):*","[O](-*)-*","[O]1-*-*-1 3-membered ring","[O]=*","[OH]-*","[O-]-*","[o](:*):*","[S](-*)-*","[S]=*","[S](-*)(-*)=*","[S](-*)(-*)(=*)=*","[SH]-*","[s](:*):*","[s](=*)(:*):*","[P](-*)(-*)-*","[P](-*)=*","[P](-*)(-*)(-*)=*","[PH](-*)(-*)=*"]),et.yG=j(F(o.P7,1),w0,6,15,[3.240000009536743,12.359999656677246,23.790000915527344,11.680000305175781,13.600000381469727,TD,12.029999732971191,21.940000534057617,23.850000381469727,26.020000457763672,0,TD,4.360000133514404,4.440000057220459,13.970000267028809,16.610000610351562,25.59000015258789,27.639999389648438,12.890000343322754,4.409999847412109,4.929999828338623,8.390000343322754,15.789999961853027,4.099999904632568,3.880000114440918,14.140000343322754,9.229999542236328,12.529999732971191,17.06999969482422,20.229999542236328,23.059999465942383,13.140000343322754,25.299999237060547,32.09000015258789,19.209999084472656,8.380000114440918,38.79999923706055,28.239999771118164,21.700000762939453,13.59000015258789,34.13999938964844,9.8100004196167,23.469999313354492]);},et.AG=function(t,A){var e,r;for(e=et.EG(t),r=0;r<et.yG.length;r++)e[r]!=0&&et.vG(A,""+e[r]+" * "+et.yG[r]+"   AtomType: "+et.xG[r],2);},et.BG=function(t){var A,e,r;for(A=et.EG(t),r=0,e=0;e<et.yG.length;e++)r+=A[e]*et.yG[e];return r;},et.CG=function(t){var A;return A=new et.wG(),U.sj(A.a,new et.tG("The polar surface area prediction is based on an atom-type based",2)),U.sj(A.a,new et.tG("increment system, published by P. Ertl, B. Rohde, P. Selzer",2)),U.sj(A.a,new et.tG("in J. Med. Chem. 2000, 43, 3714-3717",2)),U.sj(A.a,new et.tG(VP,2)),et.AG(t,A),A;},et.DG=function(t,A){var e;switch(t.C[A]){case 7:if(A<t.f&&n.ot(t.p,A)){if(t.s[A]==0){if(t.e[A]-n.tp(t,A)+n.op(t,A)==0){if(n.tp(t,A)==2)return 18;for(e=0;e<n.tp(t,A);e++)if(!n.Up(t,t.k[A][e]))return 20;return 19;}else return 22;}else if(t.s[A]==1)if(t.e[A]-n.tp(t,A)+n.op(t,A)==0){for(e=0;e<n.tp(t,A);e++)if(!n.Up(t,t.k[A][e]))return n.Bl(t,t.i[A][e])<0?21:24;return 23;}else return 25;}else if(t.s[A]==0)switch(t.e[A]-n.tp(t,A)+n.op(t,A)){case 0:switch(t.o[A]){case 0:return (t.p&&A<t.f?n.dt(t.p,A):0)==3?5:0;case 1:return 1;case 2:return 2;}break;case 1:switch(t.o[A]){case 0:return (t.p&&A<t.f?n.dt(t.p,A):0)==3?7:6;case 1:return 8;}break;case 2:return 9;}else if(t.s[A]==1)switch(t.e[A]-n.tp(t,A)+n.op(t,A)){case 0:switch(t.o[A]){case 0:return 10;case 1:return et.FG(t,A)?3:11;case 2:return t.n[A][0]==2?et.FG(t,A)?4:et.yG.length+1:12;}break;case 1:switch(t.o[A]){case 0:return 13;case 1:return 14;}break;case 2:return t.o[A]==0?15:16;case 3:return 17;}return et.yG.length+1;case 8:if(A<t.f&&n.ot(t.p,A)){if(t.s[A]==0)return 31;}else {if(t.s[A]==0)return t.o[A]>0?28:n.tp(t,A)==1?29:(t.p&&A<t.f?n.dt(t.p,A):0)==3?27:26;if(t.s[A]==-1)return n.tp(t,A)==1&&n.Bl(t,t.i[A][0])>0?28:30;}return et.yG.length+1;case 15:if(t.s[A]==0){if(t.e[A]-n.tp(t,A)+n.op(t,A)==0){if(n.tp(t,A)==3&&t.o[A]==0)return 39;if(n.tp(t,A)==2&&t.o[A]==1)return 40;if(n.tp(t,A)==4&&t.o[A]==1)return 41;}else if(t.e[A]-n.tp(t,A)+n.op(t,A)==1&&n.tp(t,A)==3&&t.o[A]==1)return 42;}return et.yG.length+1;case 16:if(t.s[A]==0){if(A<t.f&&n.ot(t.p,A))return n.tp(t,A)==2?37:38;if(t.e[A]-n.tp(t,A)+n.op(t,A)==0){if(n.tp(t,A)==2&&t.o[A]==0)return 32;if(n.tp(t,A)==1&&t.o[A]==1)return 33;if(n.tp(t,A)==3&&t.o[A]==1)return 34;if(n.tp(t,A)==4&&t.o[A]==2)return 35;}else if(t.e[A]-n.tp(t,A)+n.op(t,A)==1&&n.tp(t,A)==1)return 36;}return et.yG.length+1;}return et.yG.length;},et.EG=function(t){var A,e;for(e=B(o.Q7,H,6,et.yG.length+2,15,1),n.bw(t,7),A=0;A<t.f;A++)++e[et.DG(t,A)];return e;},et.FG=function(t,A){var e;for(e=0;e<t.j[A];e++)if(n.Bl(t,t.i[A][e])<0)return  true;return  false;},et.GG=function(t){var A,e;for(e=0,A=0;A<t.a.q;A++)(n.Tl(t.a,A)==7||n.Tl(t.a,A)==8)&&++e;return e;},et.HG=function(t){var A,e;for(e=0,A=0;A<t.a.q;A++)(n.Tl(t.a,A)==7||n.Tl(t.a,A)==8)&&n.No(t.a,A)>0&&++e;return e;},et.IG=function(t){var A;try{return et.gG((new et.iG(),t.a));}catch(e){if(e=C1(e),Zt(e,19))return A=e,h.MB(A,(h.Pkb(),h.Okb),""),-999;throw st(e);}},et.JG=function(t){this.a=t,n.lq(t);},x(278,1,{}),o.dab=N(278),et.MG=function(){et.MG=f1,et.KG=j(F(o.R7,1),e0,6,14,[524290,524292,524297,524301,589826,589827,589828,589829,589832,589833,589837,589838,688130,688131,688132,688137,688141,819202,819203,819204,819205,819208,819209,1114114,1114115,1114116,1114120,1212420,1343492,1343496,1638402,1638403,17367042,17367043,17367044,17367045,17367048,17367049,17367053,17367054,17465346,1074266116,1208483842,1208483844,1208549378,1208549379,1208549380,1208549384,1208549388,1409810434,1409875970,1409875971,1409875972,1409875976,1409974274,1409974280,1409974284,1678311426,1678311427,1678311428,1678311432,1678311436,1678409730,1678409731,1678409732,1678540802,1678540803,1678540804,1678540808,2282291202,2282291203,2282389506,2282389507,2282520578,2282520579,2483617794,2483617795,2483716098,2483716099,2752675842,3356033026,3557359618,3557457922,3557588994,34510798850,34510798851,34510798852,34510798856,34712125442,34712125443,34712125444,34712125448,34712223746,34712223747,34712223748,34980560898,34980560899,34980659202,35568287746,35568287747,35568287748,35568287752,35568386050,35568386051,35568386052,35568517122,35568517123,35568517124,35568517128,35568812034,35568812035,35568910338,35568910339,35568910350,35569336322,35585064962,35585064963,35585064964,35585064968,35769548802,35769614338,35769712642,35769843714,35786391554,d9,m9,v9,F9,j9,p9,M9,R9,C9,S9,H9,E9,q9,3437652377603,V9,I9,4674334294019,O9,x9,4674602827778,N9,G9,z9,J9,5086651154435,K9,5086651252739,Z9,5086651252744,Y9,5086919589896,X9,5086919819266,_9,W9,$9,5636675633154,{l:589826,m:73760,h:4},{l:589827,m:73760,h:4},{l:688130,m:73760,h:4},{l:819202,m:73760,h:4},{l:65538,m:73764,h:4},{l:65539,m:73764,h:4},{l:589826,m:172064,h:4},{l:589827,m:172064,h:4},{l:688130,m:172064,h:4},{l:819202,m:172064,h:4},{l:819203,m:172064,h:4},{l:65538,m:172068,h:4},{l:65539,m:172068,h:4},{l:589826,m:172112,h:4},{l:589827,m:172112,h:4},{l:688130,m:172112,h:4},{l:819202,m:172112,h:4},{l:65538,m:172116,h:4},{l:163842,m:172116,h:4},{l:589826,m:303136,h:4},{l:688130,m:303136,h:4},{l:819202,m:303136,h:4},{l:65538,m:303140,h:4},{l:589826,m:303184,h:4},{l:688130,m:303184,h:4},{l:819202,m:303184,h:4},{l:65538,m:303188,h:4},a9,tf,Af,nf,ef,rf,Pf,{l:688130,m:590224,h:4},{l:819202,m:590224,h:4},ff,Df,uf,Qf,of,lf,Lf,sf,{l:688135,m:590416,h:4},Uf,{l:1212430,m:590416,h:4},{l:65538,m:598052,h:4},{l:65538,m:598100,h:4},{l:65539,m:598100,h:4},{l:163842,m:598100,h:4},{l:163843,m:598100,h:4},yf,Tf,hf,{l:688131,m:598304,h:4},cf,bf,wf,gf,Bf,{l:589826,m:688672,h:4},kf,df,mf,vf,{l:589827,m:590112,h:336},Ff,jf,pf,Mf,Rf,Cf,Sf,Hf,Ef,{l:688130,m:819536,h:400},qf,Vf,If,{l:589831,m:590112,h:592},Of,xf,Nf,Gf,zf,Jf,{l:819207,m:819600,h:592},Kf,Zf,Yf,{l:589832,m:1212816,h:592},{l:688136,m:1212816,h:592},Xf,_f,Wf,$f,af,{l:589826,m:819488,h:8480},tD,AD,nD,eD,{l:819208,m:1213008,h:8480},rD,PD,{l:688130,m:590160,h:8484},{l:589826,m:598304,h:8484},iD,{l:819202,m:598304,h:8484}]),et.LG=j(F(o.P7,1),w0,6,15,[-0.1899999976158142,1.2699999809265137,-0.7009999752044678,2.690999984741211,-0.22699999809265137,Di,.10599999874830246,-0.47600001096725464,-0.44699999690055847,-0.19099999964237213,-0.3330000042915344,.0860000029206276,.24699999392032623,-0.06199999898672104,.01600000075995922,.3869999945163727,.23499999940395355,-0.4320000112056732,-0.902999997138977,.38999998569488525,.5809999704360962,4.52400016784668,-0.6349999904632568,-0.2800000011920929,.7699999809265137,-0.05000000074505806,1.0870000123977661,.19200000166893005,.19599999487400055,-0.5199999809265137,.5419999957084656,.3630000054836273,.7919999957084656,.5920000076293945,.9639999866485596,QD,-0.6850000023841858,-0.3149999976158142,-0.4129999876022339,-0.5950000286102295,.2199999988079071,-1.4320000410079956,-2.253999948501587,.4399999976158142,-0.27000001072883606,-0.13300000131130219,-0.26899999380111694,.2669999897480011,.5720000267028809,-0.5680000185966492,.17399999499320984,-0.1850000023841858,-0.23499999940395355,hD,cD,-0.34200000762939453,-0.3479999899864197,-0.43700000643730164,-0.8040000200271606,-0.41200000047683716,-0.2150000035762787,-0.625,-0.8309999704360962,.4970000088214874,-0.4309999942779541,-1.3309999704360962,.5070000290870667,-0.6320000290870667,-0.5989999771118164,-0.15600000321865082,-0.3529999852180481,-0.164000004529953,-0.4410000145435333,-0.4970000088214874,-1.059999942779541,.11500000208616257,-0.22499999403953552,-0.15399999916553497,-0.03099999949336052,-1.5740000009536743,-1.093000054359436,sD,-0.44999998807907104,-0.5559999942779541,cD,2.384000062942505,1.75,-1.6660000085830688,-1.065999984741211,1.3270000219345093,.8029999732971191,-1.5049999952316284,-2.5369999408721924,hD,.14900000393390656,.5210000276565552,2.9049999713897705,-0.25200000405311584,.8600000143051147,.3610000014305115,.40299999713897705,.004999999888241291,1.1460000276565552,.9359999895095825,-0.30000001192092896,.20900000631809235,-0.5830000042915344,-0.024000000208616257,-0.009999999776482582,.6110000014305115,.4860000014305115,.8619999885559082,-0.03500000014901161,-0.5960000157356262,1.1610000133514404,1.6469999551773071,.843999981880188,.125,.1420000046491623,-0.17100000381469727,.44200000166893005,.08799999952316284,3.065999984741211,1.6519999504089355,bD,-0.017999999225139618,.023000000044703484,.0729999989271164,.2540000081062317,.5540000200271606,.5950000286102295,-0.4059999883174896,-0.6370000243186951,-0.17399999499320984,-0.10100000351667404,-0.5429999828338623,-2.4059998989105225,-3.2920000553131104,-0.05299999937415123,-0.19300000369548798,SP,-1.2610000371932983,-0.656000018119812,-0.7300000190734863,-0.9380000233650208,.12800000607967377,1.1540000438690186,.24199999868869781,-0.5289999842643738,-0.27799999713897705,-0.8019999861717224,.9120000004768372,-1.38100004196167,.46299999952316284,1.0740000009536743,-0.628000020980835,-0.9620000123977661,-1.8320000171661377,-1.4989999532699585,-2.115999937057495,-2.2070000171661377,-1.3170000314712524,2.500999927520752,-0.8489999771118164,lD,-0.621999979019165,2.121999979019165,-2.2260000705718994,.9129999876022339,-2.259000062942505,-1.25,1.3940000534057617,-1.4019999504089355,2.072999954223633,-2.9570000171661377,.29100000858306885,-3.4760000705718994,-2.7269999980926514,-3.131999969482422,-2.119999885559082,-0.7250000238418579,-0.296999990940094,.08299999684095383,.34700000286102295,-1.4249999523162842,-1.659999966621399,-1.281999945640564,-1.2649999856948853,.718999981880188,.1379999965429306,1.3020000457763672,.859000027179718,1.3589999675750732,.6589999794960022,-0.9399999976158142,.8999999761581421,.3190000057220459,-2.571000099182129,1.1089999675750732,.972000002861023,1.652999997138977,2.6019999980926514,.7289999723434448,1.065999984741211,1.0670000314712524,-0.3109999895095825,.03099999949336052,bD,-0.6809999942779541,-1.2580000162124634,Ei,-3.0959999561309814,-0.2280000001192093,1.9329999685287476,.11900000274181366,2.1080000400543213,.11299999803304672,1.628000020980835,1.3079999685287476,3.3359999656677246,.7540000081062317,-0.4650000035762787,-0.3970000147819519,.07699999958276749,-0.4790000021457672,-0.15299999713897705,.14100000262260437,2.134999990463257,.23399999737739563,.460999995470047,.6700000166893005,-0.3610000014305115,-1.0390000343322754,-0.4830000102519989,.13699999451637268,-0.7680000066757202,-0.5109999775886536,3.4240000247955322,-0.8550000190734863,-0.5849999785423279,-1.5670000314712524,3.3429999351501465,1.840000033378601,.3889999985694885,1.121999979019165,HP,1.3350000381469727,.3659999966621399,-0.5569999814033508,.4320000112056732,.20399999618530273,.8820000290870667,.4659999907016754,-0.4580000042915344,.40400001406669617,.6570000052452087,1.1150000095367432,1.9759999513626099,1.7860000133514404,-0.035999998450279236,-1.0499999523162842,1.0449999570846558,.04399999976158142,1.0329999923706055,-1.0800000429153442,2.5390000343322754,2.234999895095825,jP,3.121000051498413,3.931999921798706,2.75]);},et.NG=function(t){var A,e,r,P;for(r=-0.5299999713897705,n.lq(t),n.bw(t,7),A=0;A<t.f;A++){P=-1;try{P=n.ff(t,A,2144);}catch(i){if(i=C1(i),!Zt(i,19))throw st(i);}for(e=0;e<et.LG.length;e++)if(S1(et.KG[e],P)){r+=et.LG[e];break;}}return r;},et.OG=function(t){var A,e,r,P,i,D,u;if(r=new et.wG(),U.sj(r.a,new et.tG("Solubility values are estimated applying an atom-type based increment system.",2)),U.sj(r.a,new et.tG(UD,2)),U.sj(r.a,new et.tG(VP,2)),U.sj(r.a,new et.tG("Base value = -0.530",2)),e=B(o.Q7,H,6,et.LG.length,15,1),t)for(n.lq(t),n.bw(t,7),A=0;A<t.f;A++){u=-1;try{u=n.ff(t,A,2144);}catch(Q){if(Q=C1(Q),!Zt(Q,19))throw st(Q);}for(D=0;D<et.LG.length;D++)if(S1(et.KG[D],u)){++e[D];break;}}for(P=new o0.Hnb("#0.000"),i=0;i<et.LG.length;i++)e[i]!=0&&et.vG(r,""+e[i]+" * "+o0.Enb(P,et.LG[i])+"   AtomType: 0x"+h.Djb(et.KG[i]),2);return r;},et.VG=function(){et.VG=f1,et.PG=j(F(o.Ocb,1),yt,2,6,["mutagenic","tumorigenic","irritant","reproductive effective"]),et.QG=j(F(o.Ocb,1),yt,2,6,["Mutagenicity","Tumorigenicity","Irritating effects","Reproductive effects"]);},et.WG=function(t,A){var e,r,P,i;if(!et.SG)return 0;if(et.UG[A].bc(n.rg(new n.Ng(t)))!=-1)return 3;for(i=new n.ju(1),e=new n.tw(),P=0;P<et.RG[A].a.length;P++)if(n.Tq(new n._q(false),e,U.xj(et.RG[A],P)),n.cu(i,t),n._t(i,e),n.Pt(i,1,i.c)>0)return 3;for(r=0;r<et.TG[A].a.length;r++)if(n.Tq(new n._q(false),e,U.xj(et.TG[A],r)),n.cu(i,t),n._t(i,e),n.Pt(i,1,i.c)>0)return 2;return 1;},et.XG=function(t,A){var e,r,P,i,D,u,Q;if(Q=new et.wG(),!et.SG)return U.sj(Q.a,new et.tG("Toxicity predictor not properly initialized.",2)),Q;if(D=n.rg(new n.Ng(t)),et.UG[A].bc(D)!=-1)return et.vG(Q,"This molecule is known to be "+et.PG[A]+":",2),U.sj(Q.a,new et.tG(D,1)),Q;for(u=new n.ju(1),e=false,r=new n.tw(),i=0;i<et.RG[A].a.length;i++)n.Tq(new n._q(false),r,U.xj(et.RG[A],i)),n.cu(u,t),n._t(u,r),n.Pt(u,1,u.c)>0&&(e||et.vG(Q,"High-risk fragments indicating "+et.QG[A]+":",2),e=true,et.vG(Q,U.xj(et.RG[A],i),1));for(e=false,P=0;P<et.TG[A].a.length;P++)n.Tq(new n._q(false),r,U.xj(et.TG[A],P)),n.cu(u,t),n._t(u,r),n.Pt(u,1,u.c)>0&&(e||et.vG(Q,"Medium-risk fragments indicating "+et.QG[A]+":",2),e=true,et.vG(Q,U.xj(et.TG[A],P),1));return Q.a.a.length==0&&et.vG(Q,"No indication for "+et.QG[A]+" found.",2),Q;},et.YG=function(t){var A,e,r,P;for(P=new Rt.ehb(new Rt.jhb((e=_t.cBb(t),q1.Cnb(),e))),r=new n.Zv();;)try{if(A=Rt.dhb(P),A!=null)n.Xv(r,A);else break;}catch(i){if(i=C1(i),Zt(i,75))break;throw st(i);}return r;},et.ZG=function(t){var A,e,r,P;for(P=new Rt.ehb(new Rt.jhb((r=_t.cBb(t),q1.Cnb(),r))),A=new U.Mj();;)try{if(e=Rt.dhb(P),e!=null)w.iAb(A.a,e);else break;}catch(i){if(i=C1(i),Zt(i,75))break;throw st(i);}return A;},et.$G=function(){if(et.VG(),!et.SG)try{et.UG=B(o.P8,o1,102,4,0,1),et.RG=B(o.vdb,o1,20,4,0,1),et.TG=B(o.vdb,o1,20,4,0,1),et.RG[0]=et.ZG("/resources/toxpredictor/m1.txt"),et.RG[1]=et.ZG("/resources/toxpredictor/t1.txt"),et.RG[2]=et.ZG("/resources/toxpredictor/i1.txt"),et.RG[3]=et.ZG("/resources/toxpredictor/r1.txt"),et.TG[0]=et.ZG("/resources/toxpredictor/m2.txt"),et.TG[1]=et.ZG("/resources/toxpredictor/t2.txt"),et.TG[2]=et.ZG("/resources/toxpredictor/i2.txt"),et.TG[3]=et.ZG("/resources/toxpredictor/r2.txt"),et.UG[0]=et.YG("/resources/toxpredictor/m3.txt"),et.UG[1]=et.YG("/resources/toxpredictor/t3.txt"),et.UG[2]=et.YG("/resources/toxpredictor/i3.txt"),et.UG[3]=et.YG("/resources/toxpredictor/r3.txt"),et.SG=!0;}catch(t){if(t=C1(t),Zt(t,19))h.Pkb();else throw st(t);}},x(284,1,{},et.$G),et.SG=false,o.eab=N(284),At._G=function(t,A){U.sj(t.a,A);},At.aH=function(t,A,e){U.rj(t.a,e,A);},At.bH=function(t,A){U.sj(t.f,A),t.d=-1;},At.cH=function(t,A,e){U.rj(t.f,e,A),t.d=-1;},At.dH=function(t,A){U.sj(t.g,A),t.d=-1;},At.eH=function(t,A,e){U.rj(t.g,e,A),t.d=-1;},At.fH=function(t){w.kAb(t.g.a,0),w.kAb(t.f.a,0),w.kAb(t.a.a,0),t.b=null,t.d=-1;},At.gH=function(t){var A,e,r;for(r=new U.rrb(t.g);r.a<r.c.a.length;)if(A=U.qrb(r),A.K)return  true;for(e=new U.rrb(t.f);e.a<e.c.a.length;)if(A=U.qrb(e),A.K)return  true;return  false;},At.hH=function(t){var A,e,r,P;for(e=0,A=0,r=0;r<t.g.a.length+t.f.a.length;r++)P=r<t.g.a.length?U.xj(t.g,r):U.xj(t.f,r-t.g.a.length),P.r!=0&&(e+=P.r,A+=n.Wl(P,P.q,P.r,(n.Rk(),n.Qk))*P.r);return e==0?(n.Rk(),n.Rk(),n.Qk):A/e;},At.iH=function(t,A){return U.xj(t.a,A);},At.jH=function(t){var A,e,r;if(t.d!=-1)return t.d;for(t.d=0,e=0;e<t.g.a.length+t.f.a.length;e++)for(r=e<t.g.a.length?U.xj(t.g,e):U.xj(t.f,e-t.g.a.length),A=0;A<r.q;A++)t.d<d.Math.abs(r.w[A])&&(t.d=d.Math.abs(r.w[A]));return t.d;},At.kH=function(t){var A,e,r,P;if(e=new At.zH(),t.g.a.length!=0){for(P=new n.vw(U.xj(t.g,0)),A=1;A<t.g.a.length;A++)n.Yk(P,U.xj(t.g,A));U.sj(e.g,P),e.d=-1;}if(t.f.a.length!=0){for(r=new n.vw(U.xj(t.f,0)),A=1;A<t.f.a.length;A++)n.Yk(r,U.xj(t.f,A));U.sj(e.f,r),e.d=-1;}return e;},At.lH=function(t,A){return A<t.g.a.length?U.xj(t.g,A):U.xj(t.f,A-t.g.a.length);},At.mH=function(t){return t.g.a.length+t.f.a.length;},At.nH=function(t){return t.e==null?"":t.e;},At.oH=function(t,A){return U.xj(t.f,A);},At.pH=function(t,A){return U.xj(t.g,A);},At.qH=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p;if(t.g.a.length==0||t.f.a.length==0||(T=At.jH(t),T==0))return null;for(y=B(o.Q7,E1,7,t.f.a.length,0,2),i=0;i<t.f.a.length;i++)for(g=U.xj(t.f,i),n.bw(g,15),y[i]=B(o.Q7,H,6,T+1,15,1),U.Drb(y[i],-1),A=0;A<g.q;A++){if(L=d.Math.abs(g.w[A]),L!=0&&y[i][L]!=-1)return null;y[i][L]=A;}for(D=B(o.vfb,Tt,6,T+1,16,1),P=0;P<t.g.a.length;P++)for(p=U.xj(t.g,P),n.bw(p,15),v=0;v<p.q;v++)if(L=d.Math.abs(p.w[v]),L!=0&&!D[L]){for(u=0;u<t.f.a.length;u++)if(c=y[u][L],c!=-1){if(g=U.xj(t.f,u),p.j[v]!=g.j[c]){D[L]=true;break;}if((p.u[v]&3)!=(g.u[c]&3)){D[L]=true;break;}for(Q=0;Q<p.j[v];Q++)if(e=n.Ll(p,p.i[v][Q]),e==0)D[L]=true;else {for(S=p.k[v][Q],r=false,l=0;l<g.j[c];l++){if(k=n.Ll(g,g.i[c][l]),k==0){D[L]=true;break;}if(k==e){if(r=true,b=g.k[c][l],(S<p.g&&(n.ut(p.p,S)||p.H[S]==8))^(b<g.g&&(n.ut(g.p,b)||g.H[b]==8))||!(S<p.g&&(n.ut(p.p,S)||p.H[S]==8))&&(n.fm(p,S)!=n.fm(g,b)||(p.F[S]&3)!=(g.F[b]&3))){D[L]=true,D[e]=true;break;}break;}}r||(D[L]=true);}}}return D;},At.rH=function(t){var A,e,r,P;for(r=new U.rrb(t.g);r.a<r.c.a.length;)if(A=U.qrb(r),A.q!=0)return  false;for(P=new U.rrb(t.f);P.a<P.c.a.length;)if(A=U.qrb(P),A.q!=0)return  false;for(e=new U.rrb(t.a);e.a<e.c.a.length;)if(A=U.qrb(e),A.q!=0)return  false;return  true;},At.sH=function(t){return t.c||At.gH(t);},At.tH=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(e=0,y=new U.rrb(t.g);y.a<y.c.a.length;)if(l=U.qrb(y),n.bw(l,1),l.K){for(A=0;A<l.f;A++)if(pt(J(l.B[A],i0),0)){if(d.Math.abs(l.w[A])!=0)return  false;}else ++e;}else e+=l.f;for(Q=new U.rrb(t.f);Q.a<Q.c.a.length;)if(D=U.qrb(Q),n.bw(D,1),D.K){for(A=0;A<D.f;A++)if(pt(J(D.B[A],i0),0)){if(d.Math.abs(D.w[A])!=0)return  false;}else --e;}else e-=D.f;if(e!=0)return  false;for(i=At.jH(t),r=B(o.vfb,Tt,6,i+1,16,1),L=new U.rrb(t.g);L.a<L.c.a.length;)for(l=U.qrb(L),A=0;A<l.f;A++)if(!l.K||S1(J(l.B[A],i0),0)){if(P=d.Math.abs(l.w[A]),r[P])return  false;r[P]=true;}for(u=new U.rrb(t.f);u.a<u.c.a.length;)for(D=U.qrb(u),n.bw(D,1),A=0;A<D.f;A++)if(!D.K||S1(J(D.B[A],i0),0)){if(P=d.Math.abs(D.w[A]),P>=i||!r[P])return  false;r[P]=false;}return  true;},At.uH=function(t){var A,e,r,P;if(t.g.a.length+t.f.a.length<=1)return  false;for(A=At.hH(t),P=B(o.Zab,R1,35,t.g.a.length+t.f.a.length,0,1),e=0;e<t.g.a.length+t.f.a.length;e++)if(P[e]=n.km(e<t.g.a.length?U.xj(t.g,e):U.xj(t.f,e-t.g.a.length),null),P[e]){for(r=0;r<e;r++)if(P[r]&&(P[e].c+P[e].b>=P[r].c&&P[e].c<=P[r].c+P[r].b||P[e].d+P[e].a>=P[r].d&&P[e].d<=P[r].d+P[r].a))return  true;if(e!=0&&P[e-1]&&(P[e].c-P[e-1].c-P[e].b>5*A||P[e].d-P[e-1].d-P[e].a>5*A))return  true;}return  false;},At.vH=function(t,A){t.b=A;},At.wH=function(t,A){var e,r,P;for(t.c=A,P=new U.rrb(t.g);P.a<P.c.a.length;)e=U.qrb(P),n.On(e,A);for(r=new U.rrb(t.f);r.a<r.c.a.length;)e=U.qrb(r),n.On(e,A);},At.xH=function(t,A){t.e=A;},At.yH=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b;for(u=At.jH(t),D=B(o.vfb,Tt,6,u+1,16,1),b=new U.rrb(t.g);b.a<b.c.a.length;)for(T=U.qrb(b),e=0;e<T.q;e++)if(P=d.Math.abs(T.w[e]),P!=0){if(D[P])throw st(new h.cC("Duplicate mapping no in reactants: "+P));D[P]=true;}for(i=B(o.vfb,Tt,6,u+1,16,1),y=new U.rrb(t.f);y.a<y.c.a.length;)for(l=U.qrb(y),e=0;e<l.q;e++)if(P=d.Math.abs(l.w[e]),P!=0){if(i[P])throw st(new h.cC("Duplicate mapping no in products: "+P));i[P]=true;}for(Q=B(o.Q7,H,6,u+1,15,1),r=0,A=1;A<=u;A++)D[A]&&i[A]&&(Q[A]=++r);if(r!=u){for(c=new U.rrb(t.g);c.a<c.c.a.length;)for(T=U.qrb(c),e=0;e<T.q;e++)n.rn(T,e,Q[d.Math.abs(T.w[e])],T.w[e]<0);for(L=new U.rrb(t.f);L.a<L.c.a.length;)for(l=U.qrb(L),e=0;e<l.q;e++)n.rn(l,e,Q[d.Math.abs(l.w[e])],l.w[e]<0);}},At.zH=function(){this.g=new U.Mj(),this.f=new U.Mj(),this.a=new U.Mj(),this.d=-1,this.c=false;},At.AH=function(t){var A,e,r,P,i,D;for(At.zH.call(this),D=t&&t.g?t.g.a.length:0,i=t&&t.f?t.f.a.length:0,A=t&&t.a?t.a.a.length:0,r=0;r<D;r++)U.sj(this.g,new n.vw(U.xj(t.g,r)));for(P=0;P<i;P++)U.sj(this.f,new n.vw(U.xj(t.f,P)));for(e=0;e<A;e++)U.sj(this.a,new n.vw(U.xj(t.a,e)));this.b=new n.gk(t.b),t.e!=null&&(this.e=t.e),this.c=t.c||At.gH(t);},At.BH=function(t,A){At.zH.call(this),t!=null&&(U.uj(this.g,new U.lpb(new U.dsb(t),0,A)),U.uj(this.f,new U.lpb(new U.dsb(t),A,t.length))),this.c=At.gH(this);},x(53,1,fn,At.zH,At.AH,At.BH),s.c=false,s.d=0,o.gab=N(53),At.CH=function(t,A,e){var r,P,i;return i=d.Math.sqrt((t.i[0].a-A)*(t.i[0].a-A)+(t.i[0].b-e)*(t.i[0].b-e)),i<8?1:(P=d.Math.sqrt((t.i[1].a-A)*(t.i[1].a-A)+(t.i[1].b-e)*(t.i[1].b-e)),P<8?2:(r=d.Math.sqrt((t.i[1].a-t.i[0].a)*(t.i[1].a-t.i[0].a)+(t.i[1].b-t.i[0].b)*(t.i[1].b-t.i[0].b)),i+P<r+5?3:0));},At.DH=function(t){var A,e;return A=Mt(t.i[1].a-t.i[0].a),e=Mt(t.i[1].b-t.i[0].b),d.Math.sqrt(A*A+e*e);},At.EH=function(t,A,e){var r,P,i,D,u,Q;return r=t.i[1].a-t.i[0].a,P=t.i[1].b-t.i[0].b,D=(t.i[0].a+t.i[1].a)/2,u=(t.i[0].b+t.i[1].b)/2,r==0?P<0^e>u:P==0?r<0^A>D:(i=-r/P,Q=(t.i[0].a+i*i*A-i*e+i*t.i[0].b)/(1+i*i),r<0^Q>D);},At.FH=function(t,A,e,r,P){t.i[0].a=A,t.i[0].b=e,t.i[1].a=r,t.i[1].b=P;},At.GH=function(){this.i=B(o.Xab,R1,55,2,0,1),this.i[0]=new ht.aO(),this.i[1]=new ht.aO(),this.a=0;},At.HH=function(t){var A,e,r,P;for(At.GH.call(this),A=0;A!=-1&&(e=w.SAb(t).indexOf('="',A),e!=-1);)if(r=(w.MAb(A+1,e,w.SAb(t).length),w.SAb(t).substr(A+1,e-(A+1))),A=w.SAb(t).indexOf('"',e+2),P=A==-1?(w.NAb(e+1,w.SAb(t).length+1),w.SAb(t).substr(e+1)):(w.MAb(e+1,A,w.SAb(t).length),w.SAb(t).substr(e+1,A-(e+1))),h.akb(r,"x1"))try{this.i[0].a=x0(P);}catch(i){if(i=C1(i),!Zt(i,40))throw st(i);}else if(h.akb(r,"y1"))try{this.i[0].b=x0(P);}catch(i){if(i=C1(i),!Zt(i,40))throw st(i);}else if(h.akb(r,"x2"))try{this.i[1].a=x0(P);}catch(i){if(i=C1(i),!Zt(i,40))throw st(i);}else if(h.akb(r,"y2"))try{this.i[1].b=x0(P);}catch(i){if(i=C1(i),!Zt(i,40))throw st(i);}},x(61,193,{193:1,61:1,4:1},At.GH,At.HH),s.Eb=function(t,A){return this.a=At.CH(this,t,A),this.a!=0;},s.Fb=function(){this.a=0;},s.Gb=function(){var t;return t=new At.GH(),t.i[0].a=this.i[0].a,t.i[0].b=this.i[0].b,t.i[1].a=this.i[1].a,t.i[1].b=this.i[1].b,t.g=this.g,t;},s.Hb=function(t,A){return At.CH(this,t,A)!=0;},s.Ib=function(t,A){var e,r,P,i,D,u,Q;R.F3(t,this.g?Lt.y4(W0,R.w3(t)):R.z3(t)),i=A?n.Ai(A,this.i[0].a):this.i[0].a,u=A?n.Bi(A,this.i[0].b):this.i[0].b,D=A?n.Ai(A,this.i[1].a):this.i[1].a,Q=A?n.Bi(A,this.i[1].b):this.i[1].b,e=D-i,r=Q-u,R.E3(t,d.Math.max(1,.019999999552965164*d.Math.sqrt(e*e+r*r))),R.p3(t,i,u,D,Q),P=new ht.iO(4),ht.dO(P,D+e/40,Q+r/40),ht.dO(P,D-e/5+r/10,Q-r/5-e/10),ht.dO(P,D-e/20,Q-r/20),ht.dO(P,D-e/5-r/10,Q-r/5+e/10),R.u3(t,new R.U3(P));},s.Jb=function(t){var A,e,r,P,i,D;return D=At.DH(this),i=d.Math.max(D/8,3),A=n.no(this.i[0].a,this.i[0].b,this.i[1].a,this.i[1].b),r=d.Math.abs(i*d.Math.cos(A)),P=d.Math.abs(i*d.Math.sin(A)),e=new ht.oO(),this.i[0].a<this.i[1].a?(e.c=this.i[0].a-r,e.b=this.i[1].a-this.i[0].a+2*r):(e.c=this.i[1].a-r,e.b=this.i[0].a-this.i[1].a+2*r),this.i[0].b<this.i[1].b?(e.d=this.i[0].b-P,e.a=this.i[1].b-this.i[0].b+2*P):(e.d=this.i[1].b-P,e.a=this.i[0].b-this.i[1].b+2*P),e;},s.Kb=function(){var t;return t=new h.Kkb(),h.Ikb(t,' x1="'+this.i[0].a+'"'),h.Ikb(t,' y1="'+this.i[0].b+'"'),h.Ikb(t,' x2="'+this.i[1].a+'"'),h.Ikb(t,' y2="'+this.i[1].b+'"'),t.a;},s.Lb=function(){return "arrow";},s.Mb=function(t){var A,e,r,P,i,D;switch(R.F3(t,R.B3(t)),this.a){case 1:R.t3(t,this.i[0].a-8,this.i[0].b-8,16);break;case 2:R.t3(t,this.i[1].a-8,this.i[1].b-8,16);break;case 3:i=At.DH(this),P=d.Math.max(i/8,3),A=n.no(this.i[0].a,this.i[0].b,this.i[1].a,this.i[1].b),e=P*d.Math.cos(A),r=-P*d.Math.sin(A),D=new ht.iO(4),ht.dO(D,this.i[0].a+e,this.i[0].b+r),ht.dO(D,this.i[1].a+e,this.i[1].b+r),ht.dO(D,this.i[1].a-e,this.i[1].b-r),ht.dO(D,this.i[0].a-e,this.i[0].b-r),R.u3(t,new R.U3(D));}},s.Ob=function(t,A){switch(this.a){case 1:this.i[0].a=this.o[0]+t-this.k,this.i[0].b=this.p[0]+A-this.n;break;case 2:this.i[1].a=this.o[1]+t-this.k,this.i[1].b=this.p[1]+A-this.n;break;default:n.oe(this,t,A);}},s.a=0,o.fab=N(61),At.SH=function(t,A){var e,r,P,i,D,u,Q,l,L;return t==null?null:(u=t,l=null,Q=null,L=null,D=null,e=h.hkb(t,X1(35)),e==-1?u=t:(u=(w.MAb(0,e,w.SAb(t).length),w.SAb(t).substr(0,e)),r=h.ikb(t,X1(35),e+1),r==-1?l=(w.NAb(e+1,w.SAb(t).length+1),w.SAb(t).substr(e+1)):(l=(w.MAb(e+1,r,w.SAb(t).length),w.SAb(t).substr(e+1,r-(e+1))),P=h.ikb(t,X1(35),r+1),P==-1?Q=(w.NAb(r+1,w.SAb(t).length+1),w.SAb(t).substr(r+1)):(Q=(w.MAb(r+1,P,w.SAb(t).length),w.SAb(t).substr(r+1,P-(r+1))),i=h.ikb(t,X1(35),P+1),i==-1?L=(w.NAb(P+1,w.SAb(t).length+1),w.SAb(t).substr(P+1)):(L=(w.MAb(P+1,i,w.SAb(t).length),w.SAb(t).substr(P+1,i-(P+1))),D=(w.NAb(i+1,w.SAb(t).length+1),w.SAb(t).substr(i+1)))))),At.TH(u,(A&1)!=0?l:null,(A&2)!=0?Q:null,(A&4)!=0?L:null,(A&8)!=0?D:null,false,null));},At.TH=function(t,A,e,r,P,i,D){var u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V;if(t==null||w.SAb(t).length==0||(V=h.hkb(t,X1(33)),V==-1))return null;for(b=false,y=V==0?1:0,p=0,Q=0,D?At.fH(D):D=new At.zH();y!=-1;)y>V&&(b=true),l=h.ikb(t,X1(32),y),V>y&&(l>V||l==-1)&&(l=V),L=l==-1?(w.NAb(y,w.SAb(t).length+1),w.SAb(t).substr(y)):(w.MAb(y,l,w.SAb(t).length),w.SAb(t).substr(y,l-y)),y=l==-1||l==w.SAb(t).length-1?-1:l+1,S=null,A!=null&&w.SAb(A).length!=0&&(l=h.ikb(A,X1(32),p),l==-1?S=(w.NAb(p,w.SAb(A).length+1),w.SAb(A).substr(p)):(S=(w.MAb(p,l,w.SAb(A).length),w.SAb(A).substr(p,l-p)),p=l+1)),u=null,e!=null&&w.SAb(e).length!=0&&(l=h.ikb(e,X1(32),Q),l==-1?u=(w.NAb(Q,w.SAb(e).length+1),w.SAb(e).substr(Q)):(u=(w.MAb(Q,l,w.SAb(e).length),w.SAb(e).substr(Q,l-Q)),Q=l+1)),q=new n._q(i),E=L==null?null:n.Sq(q,w.sAb((k=L,q1.Cnb(),k)),u==null?null:w.sAb((v=u,v)),0),S!=null&&n.Wq(q,w.sAb((g=S,q1.Cnb(),g))),b?(U.sj(D.f,E),D.d=-1):(U.sj(D.g,E),D.d=-1);if(r!=null&&w.SAb(r).length!=0&&At.vH(D,new n.hk(r)),P!=null&&w.SAb(P).length!=0){for(q=new n._q(i),T=0,c=h.hkb(P,X1(43));c!=-1;)At._G(D,n.Pq(q,(w.MAb(T,c,w.SAb(P).length),w.SAb(P).substr(T,c-T)))),T=c+1,c=h.ikb(P,X1(43),T);At._G(D,n.Pq(q,(w.NAb(T,w.SAb(P).length+1),w.SAb(P).substr(T))));}return D;},At.UH=function(t,A,e){var r,P,i,D,u,Q,l,L,y;return t==null?null:(Q=t,L=null,l=null,y=null,u=null,r=h.hkb(t,X1(35)),r==-1?Q=t:(Q=(w.MAb(0,r,w.SAb(t).length),w.SAb(t).substr(0,r)),P=h.ikb(t,X1(35),r+1),P==-1?L=(w.NAb(r+1,w.SAb(t).length+1),w.SAb(t).substr(r+1)):(L=(w.MAb(r+1,P,w.SAb(t).length),w.SAb(t).substr(r+1,P-(r+1))),i=h.ikb(t,X1(35),P+1),i==-1?l=(w.NAb(P+1,w.SAb(t).length+1),w.SAb(t).substr(P+1)):(l=(w.MAb(P+1,i,w.SAb(t).length),w.SAb(t).substr(P+1,i-(P+1))),D=h.ikb(t,X1(35),i+1),D==-1?y=(w.NAb(i+1,w.SAb(t).length+1),w.SAb(t).substr(i+1)):(y=(w.MAb(i+1,D,w.SAb(t).length),w.SAb(t).substr(i+1,D-(i+1))),u=(w.NAb(D+1,w.SAb(t).length+1),w.SAb(t).substr(D+1)))))),At.TH(Q,L,l,y,u,A,e));},At.VH=function(t,A,e){var r,P;return P=At.WH(t,A,(e&16)==0),P==null?null:(r=new h.Mkb(P[0]),e!=0&&(r.a+="#",(e&1)!=0&&P.length>1&&P[1]!=null&&h.Ikb(r,P[1])),e&=-2,e!=0&&(r.a+="#",(e&2)!=0&&P.length>2&&P[2]!=null&&h.Ikb(r,P[2])),e&=-3,e!=0&&(r.a+="#",(e&4)!=0&&P.length>3&&P[3]!=null&&h.Ikb(r,P[3])),e&=-5,e!=0&&(r.a+="#",(e&8)!=0&&P.length>4&&P[4]!=null&&h.Ikb(r,P[4])),r.a);},At.WH=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;for(L=B(o.Ocb,yt,2,t.g.a.length+t.f.a.length,6,1),b=B(o.Ocb,yt,2,t.g.a.length+t.f.a.length,6,1),i=B(o.Ocb,yt,2,t.g.a.length+t.f.a.length,6,1),Q=0;Q<t.g.a.length+t.f.a.length;Q++){if(v=Q<t.g.a.length?U.xj(t.g,Q):U.xj(t.f,Q-t.g.a.length),v.K&&Q<t.g.a.length)for(r=0;r<v.q;r++)n.vn(v,r,hA,false);if(P=new n.Ng(v),L[Q]=n.rg(P),L[Q]==null)return null;b[Q]=n.og(P),i[Q]=n.ng(P,A);}for(y=new h.Kkb(),D=new h.Kkb(),g=new h.Kkb(),l=0;l<t.g.a.length;l++){if(T=l,e)for(k="",T=-1,c=0;c<t.g.a.length;c++)h.Wjb(k,L[c])<0&&(k=L[c],T=c);l>0&&(y.a+=" ",g.a+=" ",D.a+=" "),h.Ikb(y,L[T]),h.Ikb(g,b[T]),h.Ikb(D,i[T]),L[T]="";}for(y.a+="!",t.f.a.length!=0&&t.g.a.length!=0&&(g.a+=" ",D.a+=" "),u=t.g.a.length;u<t.g.a.length+t.f.a.length;u++){if(T=u,e)for(k="",T=-1,c=t.g.a.length;c<t.g.a.length+t.f.a.length;c++)h.Wjb(k,L[c])<0&&(k=L[c],T=c);u>t.g.a.length&&(y.a+=" ",g.a+=" ",D.a+=" "),h.Ikb(y,L[T]),h.Ikb(g,b[T]),h.Ikb(D,i[T]),L[T]="";}return S=B(o.Ocb,yt,2,5,6,1),S[0]=y.a,w.SAb(g.a).length>t.g.a.length+t.f.a.length-1&&(S[1]=g.a),w.SAb(D.a).length>t.g.a.length+t.f.a.length-1&&(S[2]=D.a),t.b&&(S[3]=n.ek(t.b)),t.a.a.length!=0&&(S[4]=At.XH(t,A)),S;},At.XH=function(t,A){var e,r,P;for(P=new h.Kkb(),r=0;r<t.a.a.length;r++)w.SAb(P.a).length!=0&&(P.a+="+"),e=new n.Ng(U.xj(t.a,r)),h.Ikb(P,n.rg(e)),A&&(P.a+=" ",h.Ikb(P,n.ng(e,true)));return P.a;},At.YH=function(t,A,e,r,P){var i,D,u,Q;if((A.u[e]&3)!=0&&(A.u[e]&4)==0&&S1(J(A.B[e],hA),0)&&(u=d.Math.abs(A.w[e]),u!=0)){for(D=0;D<t.c.g.a.length;D++)for(i=At.pH(t.c,D),Q=0;Q<i.f;Q++)if(d.Math.abs(i.w[Q])==u)return !!(A.j[e]==r.j[P]||pt(J(i.B[Q],L0),0)||(i.u[Q]&l1)>>18!=(A.u[e]&l1)>>18);}return  false;},At.ZH=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c;for(T=B(o.Q7,H,6,t.f,15,1),c=At.aI(t,A,T),D=false,i=false,y=new U.Mj(),P=0;P<c;P++)n.ym(t,T[P])&&(l=n.qw(t,T[P],r),(l==1||l==2)&&(u=n.Nl(e,r[T[P]]),(u==1||u==2)&&(l==u?D=true:i=true)),Q=B(o.Q7,H,6,2,15,1),Q[0]=T[P],Q[1]=l,w.iAb(y.a,Q));if(!D||!i)for(L=new U.rrb(y);L.a<L.c.a.length;)l=U.qrb(L),Q=l[1],(Q==1||Q==2)&&i&&(Q=Q==1?2:1),n.un(e,r[l[0]],Q,true);},At._H=function(t){var A,e,r,P,i,D,u,Q,l,L,y;if(t.n==null){if(t.b)for(r=t.j+1,t.p=B(o.Q7,b0,28,t.o.length,0,3),i=0;i<t.o.length;i++){for(t.p[i]=B(o.Q7,E1,7,t.g[i].a.length,0,2),u=0;u<t.g[i].a.length;u++)t.p[i][u]=At.eI(t,i,U.xj(t.g[i],u),r);for(r+=t.o[i].f,P=At.pH(t.c,i),A=0;A<P.f;A++)pt(J(P.B[A],i0),0)||--r;}if(t.n=U1(o.Q8,[b0,GA],[74,26],0,[t.q,t.c.f.a.length],2),t.d=U1(o.Ocb,[b0,yt],[13,2],6,[t.q,t.c.f.a.length],2),t.e=B(o.vfb,Tt,6,t.q,16,1),y=new n.Zv(),L=new h.Kkb(),D=0,e=0,t.q!=0){Q=B(o.Q7,H,6,t.o.length,15,1);do{for(l=w.SAb(L.a).length,0<l?L.a=h.rkb(L.a,0,0):0>l&&(L.a+=h.xkb(B(o.N7,e0,6,-l,15,1))),i=0;i<t.c.f.a.length;i++)a1(t.n[D],i,At.bI(t,t.g,Q,i)),t.d[D][i]=n.rg(new n.Ng(t.n[D][i])),h.Ikb(L,t.d[D][i]);t.e[D]=n.Xv(y,L.a)==-1,(!t.t||!t.e[D])&&++e,++D;}while(At.hI(t,Q)&&e<t.i);}}},At.aI=function(t,A,e){var r,P,i,D,u,Q;for(u=B(o.vfb,Tt,6,t.f,16,1),e[0]=A,Q=1,u[A]=true,i=0;i<Q;i++)for(D=0;D<n.$o(t,e[i]);D++)P=n._o(t,e[i],D),((t.F[P]&64)!=0||n.fm(t,P)==2||n.Vp(t,P))&&(r=n.Zo(t,e[i],D),u[r]||(e[Q++]=r,u[r]=true));return Q;},At.bI=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt,ct,It,Et,Ht,Kt,xt,Nt,kt,Bt,c1,D1,St,gt,w1,I1,L1,G1,s1,Q1,z1,t0,p0,J1,u1,u0,h0,d0,RA,BA,nA,eA,R0,CA,K0,C0;for(K=At.oH(t.c,r),u1=new n.tw(),g=0,k=0,K0=null,Ut=0;Ut<t.o.length;Ut++){for(O=At.pH(t.c,Ut),n.bw(t.o[Ut],1),s1=U.xj(A[Ut],e[Ut]),G1=B(o.Q7,H,6,t.o[Ut].f,15,1),p=B(o.vfb,Tt,6,t.o[Ut].f,16,1),E=B(o.vfb,Tt,6,t.o[Ut].g,16,1),xt=0;xt<O.f;xt++)if(s1[xt]!=-1)if(d.Math.abs(O.w[xt])==0){for(q=s1[xt],p[q]=true,w1=0;w1<n.$o(t.o[Ut],q);w1++)E[n._o(t.o[Ut],q,w1)]=true;for(gt=n.Lo(t.o[Ut],q);gt<n.Mo(t.o[Ut],q);gt++)E[n._o(t.o[Ut],q,gt)]=true;}else G1[s1[xt]]=d.Math.abs(O.w[xt]);for(Nt=0;Nt<O.g;Nt++)if(y=O.D[0][Nt],T=O.D[1][Nt],d.Math.abs(O.w[y])!=0&&d.Math.abs(O.w[T])!=0&&(u=s1[y],Q=s1[T],u!=-1&&Q!=-1)){for(gt=0;gt<t.o[Ut].g;gt++)if(n.Zl(t.o[Ut],0,gt)==u&&n.Zl(t.o[Ut],1,gt)==Q||n.Zl(t.o[Ut],0,gt)==Q&&n.Zl(t.o[Ut],1,gt)==u){E[gt]=true;break;}}for(t0=B(o.Q7,H,6,t.o[Ut].f,15,1),kt=0;kt<t.o[Ut].f;kt++)if(!p[kt]&&(t0[kt]=n.ll(t.o[Ut],u1,kt,g,k),t.b&&n.rn(u1,t0[kt],t.p[Ut][e[Ut]][kt],false),G1[kt]!=0)){for(n.rn(u1,t0[kt],G1[kt],false),gt=0;gt<K.f;gt++)if(d.Math.abs(K.w[gt])==G1[kt]){if(n.jn(u1,t0[kt],K.s[gt]),n.wn(u1,t0[kt],K.u[gt]&48),t.f[r][gt])if(J1=n.Nl(t.o[Ut],kt),h0=J(K.B[gt],hA),nt(h0,0)==0)n.un(u1,t0[kt],3,false);else if(nt(h0,hA)==0)!K0&&(K0=new U.Mj()),U.sj(K0,h.qjb(t0[kt]));else if(J1==1||J1==2){for(ct=nt(h0,ke)==0,I1=0;I1<n.$o(t.o[Ut],kt);I1++)if(c=n.Zo(t.o[Ut],kt,I1),p[c])for(L1=0;L1<n.$o(t.o[Ut],kt);L1++)p0=n.Zo(t.o[Ut],kt,L1),!p[p0]&&p0>c&&(ct=!ct);ct&&(J1=J1==1?2:1),n.un(u1,t0[kt],J1,false);}else n.un(u1,t0[kt],0,false);break;}}for(Bt=0;Bt<t.o[Ut].g;Bt++)E[Bt]||n.nl(t.o[Ut],u1,Bt,g,k,t0,true);g=n.cn(u1,1),k=n.cn(u1,2);}for(z1=B(o.Q7,H,6,K.f,15,1),c1=0;c1<K.f;c1++)if(G1=d.Math.abs(K.w[c1]),G1==0)z1[c1]=n.ll(K,u1,c1,g,k);else for(gt=0;gt<u1.q;gt++)if(d.Math.abs(u1.w[gt])==G1){z1[c1]=gt;break;}if(t.r)for(xt=0;xt<K.f;xt++)n.sn(u1,z1[xt],true),n.zn(u1,z1[xt],K.J[xt].a),n.An(u1,z1[xt],K.J[xt].b);for(tt=0;tt<K.g;tt++)if((K.G[tt]&sA)==0&&(u0=n.ml(K,u1,tt,g,k,z1[K.D[0][tt]],z1[K.D[1][tt]],false),d0=K.G[tt]&31,d0!=0))for(G=false,n.Ln(u1,u0,31,false),Qt=n.Ll(K,K.D[0][tt]),z=n.Ll(K,K.D[1][tt]),bt=0;!G&&bt<t.c.g.a.length;bt++)for(O=At.pH(t.c,bt),ut=0;!G&&ut<O.g;ut++)it=O.D[0][ut],lt=O.D[1][ut],Dt=d.Math.abs(O.w[it]),ot=d.Math.abs(O.w[lt]),Dt!=0&&ot!=0&&(Dt==Qt&&ot==z||Dt==z&&ot==Qt)&&(G=true,C0=O.G[ut]&31,C0!=0&&(s1=U.xj(A[bt],e[bt]),BA=s1[it],nA=s1[lt],eA=n.Xo(t.o[bt],BA,nA),CA=n.im(t.o[bt],eA),R0=n.fm(t.o[bt],eA),C0&=7,d0&=7,C0==3&&d0==6?(u1.H[u0]=R0<=1?2:4,u1.T=0):C0==6&&d0==3?(u1.H[u0]=R0==3?2:1,u1.T=0):(u1.H[u0]=CA,u1.T=0)));for(n.bw(u1,1),v=false,D1=0;D1<K.f;D1++)d.Math.abs(K.w[D1])==0?(J1=n.qw(K,D1,z1),n.un(u1,z1[D1],J1,false)):t.f[r][D1]&&At.YH(t,K,D1,u1,z1[D1])&&(J1=n.qw(K,D1,z1),n.un(u1,z1[D1],J1,false),(J1==1||J1==2)&&(S=(K.u[D1]&l1)>>18,b=(K.u[D1]&l1)>>18!=1&&(K.u[D1]&l1)>>18!=2?-1:(K.u[D1]&F0)>>20,S==1?b+=g:S==2&&(b+=k),n.on(u1,z1[D1],S,b),v=true));for(v&&(g=n.cn(u1,1),k=n.cn(u1,2)),RA=B(o.vfb,Tt,6,K.q,16,1),St=0;St<K.f;St++)t.f[r][St]&&!RA[St]&&(K.u[St]&4)!=0&&S1(J(K.B[St],hA),0)&&K.j[St]==n.$o(u1,z1[St])&&At.ZH(K,St,u1,z1);for(wt=B(o.vfb,Tt,6,u1.q,16,1),Et=z1,Ht=0,Kt=Et.length;Ht<Kt;++Ht)It=Et[Ht],wt[It]=true;for(V=true;V;)for(V=false,L=0;L<u1.r;L++)u=u1.D[0][L],Q=u1.D[1][L],wt[u]&&!wt[Q]?(wt[Q]=true,V=true):wt[Q]&&!wt[u]&&(wt[u]=true,V=true);for(D=0;D<u1.q;D++)n.xn(u1,D,!wt[D]);if(n.wl(u1),!t.b)for(l=0;l<u1.q;l++)u1.w[l]=0;if(n.Ge(new n.Ve(u1),null,false),u1.T|=8,Q1=2|(t.r?8:0),X.jA(new X.CA(Q1),u1),K0)for(n.bw(u1,15),i=new U.rrb(K0);i.a<i.c.a.length;)P=U.qrb(i).a,(u1.u[P]&q0)!=0&&(n.un(u1,P,1,false),n.on(u1,P,1,g++));return u1;},At.cI=function(t){var A,e;if(A=t.q,t.t)for(e=0;e<t.q;e++)t.e[e]&&--A;return t.i>0&&A>t.i&&(A=t.i),A;},At.dI=function(t){var A,e,r;for(At._H(t),r=B(o.Q8,b0,74,At.cI(t),0,2),A=0,e=0;e<t.q&&A<t.i;e++)(!t.t||!t.e[e])&&(r[A++]=t.n[e]);return r;},At.eI=function(t,A,e,r){var P,i,D,u;for(u=B(o.Q7,H,6,t.o[A].f,15,1),U.Jrb(u,u.length,-1),i=At.pH(t.c,A),P=0;P<i.f;P++)e[P]!=-1&&(u[e[P]]=d.Math.abs(i.w[P]));for(D=0;D<u.length;D++)u[D]==-1&&(u[D]=r++);return u;},At.fI=function(t,A){var e,r,P,i;for(e=B(o.Q7,H,6,t.j[A]-n.bp(t,A),15,1),i=0,P=0;P<t.j[A];P++)r=t.i[A][P],pt(J(t.B[r],i0),0)||(e[i++]=t.C[r]);return w.lAb(e,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),e;},At.gI=function(t,A){var e,r,P,i;for(i=B(o.Q7,H,6,t.j[A]-n.bp(t,A),15,1),P=0,r=0;r<t.j[A];r++)e=t.i[A][r],pt(J(t.B[e],i0),0)||(i[P++]=d.Math.abs(t.w[e]));return w.lAb(i,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),i;},At.hI=function(t,A){var e;for(e=0;e<A.length;e++){if(A[e]<t.g[e].a.length-1)return ++A[e],true;A[e]=0;}return  false;},At.iI=function(t,A){var e,r,P,i,D,u,Q,l,L,y;for(t.j=0,P=0;P<A.g.a.length+A.f.a.length;P++)for(Q=At.lH(t.c,P),e=0;e<Q.f;e++)pt(J(Q.B[e],i0),0)?Q.w[e]=0:t.j<d.Math.abs(Q.w[e])&&(t.j=d.Math.abs(Q.w[e]));for(y=B(o.vfb,Tt,6,t.j+1,16,1),i=0;i<A.g.a.length;i++)for(L=U.xj(A.g,i),e=0;e<L.f;e++)u=d.Math.abs(L.w[e]),u!=0&&(y[u]=true);for(D=0;D<A.f.a.length;D++)for(l=U.xj(A.f,D),e=0;e<l.f;e++)u=d.Math.abs(l.w[e]),u!=0&&(y[u]?y[u]=false:l.w[e]=0);for(r=0;r<A.g.a.length;r++)for(L=U.xj(A.g,r),e=0;e<L.f;e++)u=d.Math.abs(L.w[e]),u!=0&&y[u]&&(L.w[e]=0);},At.jI=function(t,A,e){var r,P,i,D,u,Q,l,L;if(t.o[A]=e,t.n=null,P=At.pH(t.c,A),n.bu(t.s,P,t.o[A]),Q=8+(t.a?0:1),n.Pt(t.s,5,Q)==0)return t.g[A]=new U.Mj(),t.q=0,false;for(t.g[A]=t.s.F,D=t.g[A].a.length-1;D>=0;D--)for(l=U.xj(t.g[A],D),u=0;u<l.length;u++)if(l[u]!=-1&&(r=0,t.a&&P.s[u]==0&&n.Bl(t.o[A],l[u])!=0&&(L=n.pm(t.o[A],l[u]),r=n.mm(t.o[A],l[u],L)),t.k[A][u]>0&&t.k[A][u]>n.jp(t.o[A],l[u])-r)){U.Ej(t.g[A],D);break;}for(t.q=1,i=0;i<t.o.length;i++)t.q*=t.g[i]?t.g[i].a.length:0;return  true;},At.kI=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c;for(this.r=true,this.b=false,this.t=true,this.a=false,this.i=qt,this.c=t,this.o=B(o.Q8,GA,26,t.g.a.length,0,1),r=0;r<t.g.a.length;r++)n.On(U.xj(t.g,r),true),n.bw(U.xj(t.g,r),15);for(P=0;P<t.f.a.length;P++)n.bw(U.xj(t.f,P),15);for(At.iI(this,t),this.k=B(o.Q7,E1,7,t.g.a.length,0,2),i=0;i<t.g.a.length;i++)for(y=U.xj(t.g,i),this.k[i]=B(o.Q7,H,6,y.f,15,1),D=0;D<y.f;D++)if(l=d.Math.abs(y.w[D]),l!=0)for(u=0;u<t.f.a.length;u++)for(L=U.xj(t.f,u),Q=0;Q<L.f;Q++)d.Math.abs(L.w[Q])==l&&(A=(c=n.pm(y,D),c+n.nm(y,D,c,y.C[D])-n.vp(y,D)-(T=n.pm(L,Q),T+n.nm(L,Q,T,L.C[Q])-n.vp(L,Q))),this.k[i][D]=d.Math.max(A,0));for(this.f=B(o.vfb,o1,12,t.f.a.length,0,2),e=0;e<t.f.a.length;e++)for(L=U.xj(t.f,e),this.f[e]=B(o.vfb,Tt,6,L.f,16,1),D=0;D<L.f;D++)if(l=d.Math.abs(L.w[D]),l!=0)for(u=0;u<t.g.a.length;u++)for(y=U.xj(t.g,u),Q=0;Q<y.f;Q++)d.Math.abs(y.w[Q])==l&&(Lt.F4(At.gI(L,D),At.gI(y,Q))!=0||Lt.F4(At.fI(L,D),At.fI(y,Q))!=0)&&(this.f[e][D]=true);this.s=new n.iu(),this.g=B(o.vdb,o1,20,this.o.length,0,1);},At.lI=function(t){At.kI.call(this,t);},x(214,1,{},At.lI),s.a=false,s.b=false,s.i=0,s.j=0,s.q=0,s.r=false,s.t=false,o.hab=N(214),At.mI=function(t,A,e){var r,P,i;for(n.bw(A,1),P=U.xj(t.a,e),r=0;r<t.d.d.a.length;r++)i=Lt.M4(t.d,r),At.pI(i,A,P);n.vl(A);},At.nI=function(t,A){var e,r,P;if(t.f=A,n.bu(t.e,t.c,t.f),n.Pt(t.e,4,8)==0)return t.a=null,0;for(t.a=t.e.F,e=t.a.a.length-1;e>=0;e--)for(P=U.xj(t.a,e),r=0;r<P.length;r++)if(P[r]!=-1&&t.b[r]>0&&t.b[r]>n.jp(t.f,P[r])){U.Ej(t.a,e);break;}return t.a.a.length;},At.oI=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;for(n.bw(t,7),n.bw(A,7),this.c=t,this.d=new Lt.Q4(),r=0;r<t.g;r++)Lt.K4(this.d,new At.sI(t,r));for(e=0;e<A.g;e++)L=new At.sI(A,e),Q=(u=Lt.N4(this.d,L),u<0?-1:u),Q==-1?(L.i=(At.BI(),At.xI),Lt.K4(this.d,L)):At.rI(Lt.M4(this.d,Q),L);for(D=this.d.d.a.length-1;D>=0;D--)L=Lt.M4(this.d,D),L.i!=(At.BI(),At.AI)?Lt.M4(this.d,D).i==At.yI&&Lt.O4(this.d,D):L.i=At.zI;for(this.b=B(o.Q7,H,6,t.f,15,1),i=0;i<t.f;i++)for(l=0;l<A.f;l++)if(d.Math.abs(A.w[l])==d.Math.abs(t.w[i])){P=(T=n.pm(t,i),T+n.nm(t,i,T,t.C[i])-n.vp(t,i)-(y=n.pm(A,l),y+n.nm(A,l,y,A.C[l])-n.vp(A,l))),this.b[i]=P>0?P:0;break;}this.e=new n.iu();},x(285,1,{},At.oI),o.kab=N(285),At.pI=function(t,A,e){var r,P,i,D;if(r=e[t.a],P=e[t.b],t.i==(At.BI(),At.xI))return n.Wk(A,r,P,t.g),true;if(i=n.Xo(A,r,P),t.i==At.vI)return n.Mn(A,i,t.g),true;if(t.i==At.wI){if(D=n.fm(A,i)+t.f,D>=0&&D<=3)return n.Mn(A,i,(n.Rk(),D==0?16:D==1?1:D==2?386:D==3?4:D==4?32:64)),true;}else if(t.i==At.zI)return A.H[i]=512,true;return  false;},At.qI=function(t,A){return t.d!=A.d?t.d<A.d?-1:1:t.e!=A.e?t.e<A.e?-1:1:0;},At.rI=function(t,A){var e,r;t.c==A.c?t.i=(At.BI(),At.yI):rn(A.c)==1?(t.g=A.c==1?1:A.c==2?386:A.c==4?4:A.c==16?16:8,t.i=(At.BI(),At.vI)):(e=(t.c&16)!=0?0:(t.c&1)!=0?1:(t.c&2)!=0?2:3,r=(A.c&16)!=0?0:(A.c&1)!=0?1:(A.c&2)!=0?2:3,r==e?t.i=(At.BI(),At.yI):(t.f=r-e,t.i=(At.BI(),At.wI)));},At.sI=function(t,A){var e,r,P;this.a=t.D[0][A],this.b=t.D[1][A],r=n.Ll(t,this.a),P=n.Ll(t,this.b),r<P?(this.d=r,this.e=P):(this.d=P,this.e=r),this.c=t.G[A]&31,e=t.H[A]&127,e==16?this.c|=16:e==2?this.c|=2:e==4?this.c|=4:this.c|=1,this.i=(At.BI(),At.AI);},x(183,1,{183:1,32:1},At.sI),s.Sb=function(t){return At.qI(this,t);},s.mb=function(t){return this.d==t.d&&this.e==t.e;},s.a=0,s.b=0,s.c=0,s.d=0,s.e=0,s.f=0,s.g=0,o.jab=N(183),At.BI=function(){At.BI=f1,At.xI=new At.CI("CREATE",0),At.wI=new At.CI("CHANGE_DIF",1),At.vI=new At.CI("CHANGE_ABS",2),At.zI=new At.CI("REMOVE",3),At.yI=new At.CI("NO_CHANGE",4),At.AI=new At.CI("UNKNOWN",5);},At.CI=function(t,A){je.call(this,t,A);},At.DI=function(){return At.BI(),j(F(o.iab,1),y0,98,0,[At.xI,At.wI,At.vI,At.zI,At.yI,At.AI]);},x(98,56,{98:1,4:1,32:1,56:1},At.CI),o.iab=zA(98,At.DI),a.EI=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(n.bw(A,1),b=t.k,g=0,k=b.length;g<k;++g)c=b[g],l=e[c.a],L=e[c.b],y=n.Xo(A,l,L),y==-1?n.Wk(A,l,L,c.e):c.e==-1?A.H[y]=512:c.e!=-2&&n.Mn(A,y,c.e);if(u=n.vl(A),t.a.length!=0)for(n.bw(A,7),P=t.a,i=0,D=P.length;i<D;++i)r=P[i],Q=u==null?e[r]:u[e[r]],T=A.u[Q]&3,n.un(A,Q,T==1?2:1,false),n.wq(A,Q);return u;},a.FI=function(t){var A,e,r,P,i;for(i=new a.bJ(t.g,t.f),P=B(o.Q7,H,6,t.g.q,15,1),e=0;e<t.g.q;e++)P[e]=n.Ll(t.g,e);for(r=B(o.Q7,H,6,t.f.q,15,1),A=0;A<t.f.q;A++)r[A]=n.Ll(t.f,A);return -a.aJ(i,a.VI(i,P,r))*.25;},a.GI=function(t,A){var e,r,P,i,D;for(t.i=B(o.Q7,H,6,t.g.f,15,1),r=0;r<t.f.f;r++)n.Ll(t.f,r)!=0&&(t.i[A[n.Ll(t.f,r)]]=n.mw(t.f,r));if(D=B(o.Q7,H,6,t.g.q,15,1),i=n.hp(t.g,D,false,false),i>1)for(P=B(o.Q7,H,6,i,15,1),e=0;e<t.g.f;e++)t.i[e]|=P[D[e]]++<<5;},a.HI=function(t,A,e,r,P){var i,D,u,Q;if(t.j.length==0||t.n.a.length==0)return null;for(Q=B(o.vfb,o1,12,A,0,2),u=B(o.vfb,Tt,6,r,16,1),D=0;D<P.length;D++)P[D]!=-1&&n.Ll(t.f,D)==0&&(u[P[D]]=true);for(i=0;i<e.length;i++)e[i]!=-1&&n.Ll(t.g,i)==0&&(Q[e[i]]=u);return Q;},a.II=function(t,A,e){var r,P,i,D,u,Q,l;if(Q=false,n.Qo(t.g,A)==0)for(u=1;u<n.$o(t.g,A);u++)for(l=0;l<u;l++)r=n.Zo(t.g,A,u),P=n.Zo(t.g,A,l),i=n.Ll(t.g,r),D=n.Ll(t.g,P),e[i]>e[D]^r>P&&(Q=!Q);return Q;},a.JI=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O;for(this.d=t,O=At.UH(A,false,null),(O.g.a.length!=1||O.f.a.length!=1)&&h.Pkb(),O.c||At.gH(O)||h.Pkb(),this.g=U.xj(O.g,0),this.f=U.xj(O.f,0),n.bw(this.g,1),n.bw(this.f,63),l=new Lt.Q4(),this.c=B(o.Q7,H,6,this.g.f+1,15,1),this.c[0]=-1,r=0;r<this.g.f;r++)n.Ll(this.g,r)!=0&&(this.c[n.Ll(this.g,r)]=r);for(a.GI(this,this.c),V=B(o.vfb,Tt,6,this.g.g,16,1),v=0;v<this.f.g;v++)g=n.Zl(this.f,0,v),k=n.Zl(this.f,1,v),T=n.Ll(this.f,g),c=n.Ll(this.f,k),T!=0&&c!=0&&(P=this.c[T],i=this.c[c],S=n.jm(this.f,v),q=n.Xo(this.g,P,i),q==-1?Lt.K4(l,new a.LI(P,i,T,c,S)):((n.hm(this.g,q)&31)==0&&(G=n.jm(this.g,q),G!=S&&Lt.K4(l,new a.LI(P,i,T,c,S))),V[q]=true));for(Q=0;Q<this.g.g;Q++)V[Q]||(P=n.Zl(this.g,0,Q),i=n.Zl(this.g,1,Q),!n.Jm(this.g,P)&&!n.Jm(this.g,i)&&(T=n.Ll(this.g,P),c=n.Ll(this.g,i),(T!=0||c!=0)&&Lt.K4(l,new a.LI(P,i,T,c,-1))));for(this.k=Lt.P4(l,B(o.lab,{474:1,4:1,5:1,11:1,8:1},148,0,0,1)),L=0,D=0;D<this.g.f;D++)n.Ll(this.g,D)==0&&!n.Jm(this.g,D)&&++L;for(this.j=B(o.Q7,H,6,L,15,1),L=0,u=0;u<this.g.f;u++)n.Ll(this.g,u)==0&&!n.Jm(this.g,u)&&(this.j[L++]=u);for(this.n=new U.Mj(),y=B(o.vfb,Tt,6,this.f.f,16,1),e=0;e<this.f.f;e++)n.Ll(this.f,e)==0&&!n.Jm(this.f,e)&&!y[e]&&U.sj(this.n,new a.VJ(this.f,e,y));for(this.b=this.j.length==0&&this.n.a.length==0,this.a=B(o.Q7,H,6,0,15,1),b=0;b<this.f.f;b++)p=n.Nl(this.f,b),(p==1||p==2)&&(E=this.c[n.Ll(this.f,b)],E!=-1&&(K=n.Nl(this.g,E),a.II(this,b,this.c)==(K==p)&&(this.a=U.zrb(this.a,this.a.length+1),this.a[this.a.length-1]=E)));this.e=a.FI(this);},x(199,1,{199:1},a.JI),s.b=false,s.e=0,o.nab=N(199),a.KI=function(t,A){return t.c<A.c?-1:t.c>A.c?1:t.d<A.d?-1:t.d>A.d?1:0;},a.LI=function(t,A,e,r,P){e<r?(this.a=t,this.b=A,this.c=e,this.d=r):(this.a=A,this.b=t,this.c=r,this.d=e),this.e=P;},x(148,1,{148:1,32:1},a.LI),s.Sb=function(t){return a.KI(this,t);},s.a=0,s.b=0,s.c=0,s.d=0,s.e=0,o.lab=N(148),a.QI=function(){a.QI=f1,a.NI=j(F(o.Ocb,2),b0,13,0,[j(F(o.Ocb,1),yt,2,6,["e","daXJ@@PjyITuiX@`!dahJ@@SJYIMMfPB#IaLJfxP IaAhl[`#!BDpAl@AL@[@Be}aL@[@@ !B|Osp?QZR?O_}}zGze`@##"]),j(F(o.Ocb,1),yt,2,6,["g","gBa`@lde@XS@!gCA`@mDPOdku`#qbq qJt#!B[G|S_qgq !BjW}q]cga##"]),j(F(o.Ocb,1),yt,2,6,["o","gGQHDDqIUjdB!gGQHHDqAeMjhB#qbqk qfQ]#!B@AL@[@@S@Fp@Dp !BFQ{~_q|ZGvUSYp##"]),j(F(o.Ocb,1),yt,2,6,["r","gOQdEjHbAFQRBrtAaJ!gNQdEbHbABCTKPFDH#qbMwX qbM~X#!BNDm`ra?UjW~YBYX@ !Ba[zw?_x@?g~H?XO~##"]),j(F(o.Ocb,1),yt,2,6,["s","gOp`ATigujj`H!gOp`ATiVKjj`H#qnyS` qY~eP#!BTLtk^sE?BOs|]pc} !BbOvw?_y??g~H?[_}##"]),j(F(o.Ocb,1),yt,2,6,["t","gOP`@dcUZdB!gNp`@tiTMjj@`#q{ir` qkLrx#!Be`Bzr_wp?OC}|Osp !B?g~w@k_}m?vw@n[a##"]),j(F(o.Ocb,1),yt,2,6,["u","daXB`Hrn@HrPEILt`D!daXB`DJn@HRUMjV@H#IxVaLJ` IylJhPP#!B[G}l@OKyDpAl@AL@[@@ !B@Fp@DpAl@AN]?`Al@AL##"]),j(F(o.Ocb,1),yt,2,6,["Sakurai","gOQH@wAINvZ@pdcFe@x@!gOQH@wAIgJi@pdcFe@x@#qreKx qrLkx#!BDpAl@IknDw|S@Fp@ !Bb@JH?_x@b@JH?Ven##"]),j(F(o.Ocb,1),yt,2,6,["Mitsunobu","gFP`ATfRhdPp`}KEYg]d@!gFP`ATfRhd`pekL{l`#qrLk qZLn#!B@hc}b@C~@h`YM` !B@hc}b@C~@h`YM`##"]),j(F(o.Ocb,1),yt,2,6,["Aldol-Addition","gOQ@AdTAcS@[Q^crJTLES`DJsL?vH!gGQ@@dsuRAcJg@HUaX#qYEbp qYub#!BxOyBzLKg`dG~xG~{ !Bb@K~@Hc}FBIA@@##"]),j(F(o.Ocb,1),yt,2,6,["Aldol-Condensation","gOQ@AdTAcS@[Q^crJTLES`DJsL?vH!gFQ@@`rrpdlHHpipBEXf@#qYEbp q^aU#!B{ZRRqA?AQfyA@L_C !B}QFw@h`B_tnH_P##"]),j(F(o.Ocb,1),yt,2,6,["Acetal-Aldol-Addition","dmdB@@serQS@sJjfd@p`Xir\\@`j\\@aUJXK@!daxL@@[df[ZjT@qQdxACdxABjTqf@#qB@`OuX qBtM{#!B_]]}rHKWw?y}uy[~GJbBu{wWqG| !BfJK~TIa]fJJghg{`pP@##"]),j(F(o.Ocb,1),yt,2,6,["Acetal-Aldol-Condensation","dmdB@@serQS@sJjfd@p`Xir\\@`j\\@aUJXK@!gNp`CTjUiV@qQS`DJg@HUVXV@#qB@`OuX qqj{`#!B?[_}b@Jw_?{}m~[~[N@Bm?vwkN| !BfsK~yzPrw}m?rzQM##"]),j(F(o.Ocb,1),yt,2,6,["Acetal-Aldol-Condensation-Cyclization","dkLB@@ZURYUvUjljh@paHpr\\@`!didD@@yIfUXXHL@CFNS`D@#IXljNPY@@@ IXljIyA#!BbOw~_x`Bm?vH?[_}b@JH?_y?b@Jw?Xc} !BbOvH?Oy??`BH?Xa}?`C~_p##"]),j(F(o.Ocb,1),yt,2,6,["Enolester-Cleavage","gOQ`@fdscT`_Qp!gOQ`@cdTASS@P#q}Kr` q}cNP#!B@k]}mpC~@k]}mqdY !Bb@K~@Hc}BfzH@hc}##"]),j(F(o.Ocb,1),yt,2,6,["propargylEnone","gCa@@dmXFD@!gCa@@dkHD#qNT qLV#!BbOw~_?y? !B@AL@[@@S##"]),j(F(o.Ocb,1),yt,2,6,["Arndt-Eistert","daiDaJYBBHj^{HhAYMpAaA@!daiD`FzLBHPVsZl@p`@#IyHHZ[@ IzDGBi`#!B?`Bw?H`Bn]{~\\g?~@Ox !B@rzH?_y?b@JH?_n~bOt##"]),j(F(o.Ocb,1),yt,2,6,["Curtius","gO]IcVaDF[s{HhCIe@`!gN]HMWADHJfm`XP@#q~Jk` qytUX#!B@O{|b@Jw\\o{~_?x@ !Bj]y??g?~?[^G_hc}##"]),j(F(o.Ocb,1),yt,2,6,["diazomethanHomologation","gFU@lQioIIs\\AyH!gFU@CPdimXD#qbM^ qbqk#!B?X`BbFZN?`C~_p !B@AL@[@@Su[x@Dp##"]),j(F(o.Ocb,1),yt,2,6,[wD,"deD@@LdbEdSTu@FqHWSDda`JFChXIa`?tdKi@!deD@@Ldb`\\SUM@FqHPsDda`JF@XXIa`?tdHY@#qTEApX qQECf@#!BQzK~}ubbW`BEgcb]?a@gg[zO !BQzK~}ubbW`Ag{VVAQzJ~c?xP##"]),j(F(o.Ocb,1),yt,2,6,[wD,"deD@@LdbEdSTu@FqHWSDda`JFChXIa`?tdKi@!gC`@DiZDE@#qPDA@p qQf#!BmpK~_x`Bm?tAs[]}?`BH_[_| !B_vp@[G|S##"]),j(F(o.Ocb,1),yt,2,6,["yne-Metathesis","daX@@LdPLWWPAlRXwQIHXLa`ZFChXO}IL[`!daX@@LdPLWWPAlRXwQIHXLa`ZFChXO}IL[`#qT@q@ qt@Q@#!BZmoi@Fjo|SFe|IkGiU@ !BZmoi@Fjo|SFe|IkGiU@##"]),j(F(o.Ocb,1),yt,2,6,["ene-yne-Metathesis","dcd@@LdPLPLWSSPAlrXwKlVRFCHXFa`zFAXXMa`?udqnWP!dcT@@LdbbplTsTt@[MeMr{Ela`jFAhXNa`VFCXXO}[J[et#qe@N@S@ qeHP@s@#!B?g?~@Oy?^gG}bOvw?H`E@PJw@hc}mp !B?`BH?[_}mpJH@oy??`AuC`Jw@hc}mp##"]),j(F(o.Ocb,1),yt,2,6,["Alkyne-Cyclisation","gG`@DcO|CFAFC`@!gFp@DiTt@@CFAFC`@#qi\\V qiSt#!B_y[qA`Biu^zV@@ !B?g~w@k_}m?vw@`##"]),j(F(o.Ocb,1),yt,2,6,["Elimination-Claisen","gNp`AldTQji@~a`!gGQ@@djmRA@#qtHUX qtSi#!Bm?vH?[\\B?g~H@hc} !B@AL@[@@S@Fp@Dp##"]),j(F(o.Ocb,1),yt,2,6,["imineFormationAzaCope","daZH@LAIMUjd@pRL@!daZH@HAAn]jd@p`@#IGfaLJ` IFDzfK@#!BDpAl@IkqDpAl@AL@[@@ !BFaFw@h`BbOw~@H`BbOt##"]),j(F(o.Ocb,1),yt,2,6,["didehydroCopeWithAromatisation","gNp@Di]ejDcjcr|wK`!gOp@DjWkB@@H#qrLkx q\\oQp#!B?g~H?K_}bGvH?H`B !Bm?w~@Hc}mpJw@ox@##"]),j(F(o.Ocb,1),yt,2,6,["symAldolNitrogenRing","dovJ@GBfttf\\v\\qjViPCADGbDodnGp!doNJ@JCSmtefWTCaYjje@H#IlZXi]]yL~C IqMVCzaIim?#!BQtl_riY?Qtl_rfuvNCQ`uZd@NCQ`uVVu}?sA]P !B?`BH@ox@bGvH@k\\Bb@JH_Xa}b@K~_rYltUr|W@##"]),j(F(o.Ocb,1),yt,2,6,["Diels-Alder","gFP@LdPLjA@!gFp@DiTujhB#qiZf qMSf#!B?_C}}?spIPFV@@ !B?g~w@k_}m?vw@`##"]),j(F(o.Ocb,1),yt,2,6,["Claisen-RA","gGQ@@eKtRA@!gGQ@@djqRA@#qMsT qM\\V#!B_qL@Dw}l@Fp@Dp !B@AL@[@@S@Fp@Dp##"]),j(F(o.Ocb,1),yt,2,6,["Cope","gGQ@DeZmRAbhcApIF@P@!gGQ@HeZmRAbhc@HIFC`@#qkNT qi\\V#!B_vp@[@@S@Fp@Dp !B_vp@[@@S@Fp@Dp##"]),j(F(o.Ocb,1),yt,2,6,["OxyCope","gNq@@dr}SHFD@!gNq@@djkUHD#qynZ` qykbp#!B?g~w?Xc}mpJH@hc} !B@Fp@DpAl@AL@[@@S##"]),j(F(o.Ocb,1),yt,2,6,["Vinylcyclopropane","gKP@DklVj@`!gKP@Di\\Vj@`#qaVh qTqh#!Bm?vH?PC~?K\\ !B?g|_Fa}eTv\\##"]),j(F(o.Ocb,1),yt,2,6,["Furan-CH2-OH-Rearrangement","gOp`@tiguif`H!gOp`@tfWMZZ`H#qZna@ qZtM@#!BTLtk^sE?BOs|]pc} !BBOuPtdy?UGm@V]Ho##"]),j(F(o.Ocb,1),yt,2,6,["rearrangement1032","gOp`ATieMji`H!gOp`ATeekZj`H#qaSnx qa]~P#!BTLtk^pc|LW?|]pc} !BBOpH?UCRSg}T^tAY##"]),j(F(o.Ocb,1),yt,2,6,["Pinacol-Rearrangement","gNp`ATffjj@pPh!gOP`@tfXZhCAA`#qb^ix qb^oH#!B@k^H@k_}@k_~@Hc} !BbOvH@oy??`BH?PFf##"]),j(F(o.Ocb,1),yt,2,6,["1,3-WaterEliminationWith12Shift","gJQ@@dmU@_SP!gKa@@`vtA}A@#qiTp qjap#!BbOvH@ox@bOt !BJdE?[@Al@AL##"]),j(F(o.Ocb,1),yt,2,6,["Epoxydation","gB``ADcdCB@!gC``AhtUPGtt@#qqb qtQ#!BjW}Y\\YX@ !B?g~w?^Va##"]),j(F(o.Ocb,1),yt,2,6,["oxydativePropargylAmine13Shift","gKi@HDEZpLHOQP!gJY@BDeVXB#qMr` qNTh#!BqiXTy{U?mW| !B@Fp@DpAl@AL##"]),j(F(o.Ocb,1),yt,2,6,["Baeyer-Villiger","gFQ`@[dTAZ`LHP!gFQ`@jdrMPGtl@#qrak qrlK#!B_?{}mwvHs^FVP@ !BbOvH@oy?bOuQzP##"]),j(F(o.Ocb,1),yt,2,6,[gD,"gGa@@dYs@XHFJIBDQbUeHPbLRl@!gFQ@@eNUPFJIBDQbUeHPbLRls@`#qbq] qfQk#!B@AOIDW}l@tD@Dp !B_qL@Dw}l_qNcDP##"]),j(F(o.Ocb,1),yt,2,6,[gD,"gBa@@d\\`XP@!gJQ@@eOU@XpdHQFIVY`P#qbq qfQ@#!B@AOIDW}l !B_qL@Dw}l_qL##"]),j(F(o.Ocb,1),yt,2,6,["Hantzsch Thiazol","daZHPDp@chaIMefh@ppDzTD~hYmC^bhbcPp]dQbUg~pp!gKXHL@aJWFe`H#qNPe@ qNj`#!BvuK[KUxv_yS[k_zhvuH !BTqa`FbpX?`@##"]),j(F(o.Ocb,1),yt,2,6,["Oxadiazole","gOX`BEdTASW@XQ@!gOu@HPeKNMKTA@#qrDMX qpULX#!BmpK~@K_}Mlx@?`C~ !BZ?`C}v|m_rYR[z?\\##"]),j(F(o.Ocb,1),yt,2,6,["Imidazole","dmeHPNg@qJqLbTtATijZ@LLJnuDmhWtSDXUFC`?rIoTAP!gOt@ATieuej`H#qDPpM_@ q~ZM`#!BqvKGg_yOqvKGg_xphrGkLcz@_sD !BTMHkACD@BOw|B@QT##"]),j(F(o.Ocb,1),yt,2,6,["1,2,3-Triazole","gB`HAbIGXFDWiM@!gF|@ADeKXmT`P#QIp Q@v#!B_vpU?g}l !BTv]`YRqg?g|XK@##"]),j(F(o.Ocb,1),yt,2,6,["1,2,4-Triazole","deFD@NALbbfASUW@FD]YJZLUCAVJ}?nES@!gO|@ABeKNLuRA@#qDB@FM q@LuP#!BY?r~@F_]jDJW`j`}Iaxx[UC] !BTv]@IPqgog|hCBT_##"])]);},a.RI=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it;for(a.UI(),Q=new a.kK(),a.hK(Q,A),v=Q.j,T=Q.d,n.bw(v,1),n.bw(T,1),q=new n.iu(),k=new n.iu(),n.cu(q,v),n.cu(k,T),t.c=r1,D=null,i=null,P=0,tt=0,t.a=new h.Kkb(),e=new n.tw(),G=a.OI,K=0,O=G.length;K<O&&(V=G[K],tt++!=t.b);++K)if(n._t(q,V.g),n.au(q,V.i),n.Pt(q,6,8)!=0&&(n._t(k,V.f),n.Pt(k,2,8)!=0&&q.F.a.length<=512)){for(u=-1e4,g=U.xj(k.F,0),E=new U.rrb(q.F);E.a<E.c.a.length&&(p=U.qrb(E),!(tt++>=t.b));)n.ol(v,e),l=a.EI(V,e,p),it=a.HI(V,v.f,p,T.f,g),n.bw(e,1),r=B(o.Q7,H,6,e.f,15,1),b=B(o.Q7,H,6,T.f,15,1),a.eK(Q,e,T,r,b,it),z=Q.q-V.e,u<z&&(u=z),t.c<z&&(t.c=z,D=a.SI(r,b,l,v.f),i=b,P=Q.b);y=Q.a<=1?"":BD+Q.a+")",h.Ikb(t.a,V.d+(V.b?"_S":V.j.length==0?"_L":V.n.a.length==0?"_I":"_U")+(Lt.B4(),Lt.C4(u,5))+y+d1);}S=B(o.Q7,H,6,v.f,15,1),c=B(o.Q7,H,6,T.f,15,1),a.eK(Q,v,T,S,c,null),Qt=Q.q,t.c<=Qt&&(t.c=Qt,D=S,i=c,P=Q.b),L=Q.a<=1?"":BD+Q.a+")",h.Ikb(t.a,(Lt.B4(),"no rule:"+Lt.C4(Qt,5)+L+d1)),t.c!=r1&&a.YJ(Q,A,D,i,P);},a.SI=function(t,A,e,r){var P,i,D,u,Q;for(D=B(o.vfb,Tt,6,t.length+1,16,1),Q=B(o.Q7,H,6,r,15,1),u=0;u<r;u++)P=e==null?u:e[u],P!=-1&&(Q[u]=t[P],D[Q[u]]=true);for(i=0;i<A.length;i++)D[A[i]]||(A[i]=0);return Q;},a.TI=function(){a.QI(),this.b=qt;},a.UI=function(){var t,A;if(a.OI==null&&a.OI==null){for(t=B(o.nab,R1,199,a.NI.length,0,1),A=0;A<a.NI.length;A++)t[A]=new a.JI(a.NI[A][0],a.NI[A][1]);a.OI=t;}},x(299,1,{},a.TI),s.b=0,s.c=0,o.mab=N(299),a.VI=function(t,A,e){var r,P,i,D;for(i=B(o.Q7,H,6,t.b.f+1,15,1),U.Jrb(i,i.length,-1),P=0;P<e.length;P++)e[P]!=0&&(i[e[P]]=P);for(D=B(o.Q7,H,6,t.b.f,15,1),U.Jrb(D,D.length,-1),r=0;r<A.length;r++)D[r]=i[A[r]];return D;},a.WI=function(t,A){var e,r,P,i;return e=t.D[0][A],r=t.D[1][A],P=n.Hm(t,e),i=n.Hm(t,r),!P&&!i?A<t.g&&n.pt(t.p,A)?Ni:mP+n.fm(t,A)/10:P&&i||P&&n.Lm(t,r)||i&&n.Lm(t,e)?ce:P&&a.oK(t,r,e)||i&&a.oK(t,e,r)?Qr:P&&a.mK(t,r,e)||i&&a.mK(t,e,r)?SP:P&&r<t.f&&n.ot(t.p,r)||i&&e<t.f&&n.ot(t.p,e)?1.9500000476837158:mP;},a.XI=function(t,A){return A<t.g&&(n.ut(t.p,A)||t.H[A]==8)?0:t.H[A]&127;},a.YI=function(t,A){return A<t.g&&(n.ut(t.p,A)||t.H[A]==8)?1.5:n.fm(t,A);},a.ZI=function(t,A,e){var r,P,i;if(r=e[A],r!=-1&&n.Nl(t.a,r)!=0&&a.$I(t,A,r,e)){if(i=n.Nl(t.b,A),P=n.Nl(t.a,r),i==3){if(P==1||P==2)return 5;}else if(P==3||a._I(t,A,e)==(i==P))return 5;}return 0;},a.$I=function(t,A,e,r){var P,i,D,u;if(n.$o(t.b,A)!=n.$o(t.a,e))return  false;for(i=0;i<n.$o(t.b,A);i++){for(u=n.Zo(t.b,A,i),P=false,D=0;D<n.$o(t.a,e);D++)if(r[u]==n.Zo(t.a,e,D)){P=true;break;}if(!P)return  false;}return  true;},a._I=function(t,A,e){var r,P,i,D,u,Q,l;if(Q=false,n.Qo(t.b,A)==0)for(u=1;u<n.$o(t.b,A);u++)for(l=0;l<u;l++)r=n.Zo(t.b,A,u),P=n.Zo(t.b,A,l),i=e[r],D=e[P],i!=-1&&D!=-1&&i>D^r>P&&(Q=!Q);return Q;},a.aJ=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(y=0,D=B(o.vfb,Tt,6,t.a.f,16,1),r=A,P=0,i=r.length;P<i;++P)e=r[P],e!=-1&&(D[e]=true);for(T=B(o.vfb,Tt,6,t.a.g,16,1),k=0;k<t.b.g;k++)if(b=n.Zl(t.b,0,k),g=n.Zl(t.b,1,k),!(t.b.K&&(n.Jm(t.b,b)||n.Jm(t.b,g)))){if(u=A[b],Q=A[g],a.YI(t.b,k),u==-1||Q==-1){(u!=-1||Q!=-1)&&(y+=a.WI(t.b,k));continue;}if(L=n.Xo(t.a,u,Q),L==-1){y+=a.WI(t.b,k);continue;}T[L]=true,y+=a.XI(t.b,k)==a.XI(t.a,L)?0:1;}for(l=0;l<t.a.g;l++)t.a.K&&(n.Jm(t.a,n.Zl(t.a,0,l))||n.Jm(t.a,n.Zl(t.a,1,l)))||T[l]||(y+=a.WI(t.a,l));for(c=0;c<t.b.f;c++)n.Nl(t.b,c)!=0&&(y+=a.ZI(t,c,A));return -y;},a.bJ=function(t,A){this.b=t,this.a=A;},x(202,1,{},a.bJ),o.oab=N(202),a.cJ=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;if(v=new a.bJ(t.e,t.c),g=a.VI(v,t.f,t.d),t.a.a.length==0)return a.aJ(v,g);for(S=1,E=new U.rrb(t.a);E.a<E.c.a.length;)p=U.qrb(E),S*=a.iJ(p);if(S>4e6)return h.Pkb(),-S;for(A=0,D=B(o.Q7,H,6,t.a.a.length,15,1),c=B(o.Q7,H,6,t.a.a.length,15,1),Q=0;Q<t.a.a.length;Q++)p=U.xj(t.a,Q),c[Q]=a.lJ(p),D[Q]=A,A+=p.a;for(i=IP,P=null,r=null,b=B(o.Q7,H,6,t.a.a.length,15,1),T=t.a.a.length!=0;T;){if(q=false,t.i!=null){for(l=0;l<t.a.a.length;l++)if(a.mJ(U.xj(t.a,l),b[l],t.i)){q=true;break;}}if(!q){for(l=0;l<t.a.a.length;l++)a.hJ(U.xj(t.a,l),b[l],g);if(k=a.aJ(v,g),i<k)for(i=k,P=B(o.Q7,H,6,A,15,1),r=B(o.Q7,H,6,A,15,1),e=0,L=0;L<t.a.a.length;L++)p=U.xj(t.a,L),a.kJ(p,b[L],P,e),a.jJ(p,b[L],r,e),e+=p.a;}for(T=false,u=0;u<b.length;u++){if(++b[u],b[u]<c[u]){T=true;break;}b[u]=0;}}if(i!=IP)for(y=t.b,u=0;u<A;u++)++y,t.f[P[u]]=y,t.d[r[u]]=y;return i;},a.dJ=function(t,A,e,r,P,i){var D,u,Q,l,L,y,T,c;for(this.e=t,this.c=A,this.f=e,this.d=r,this.g=P,this.b=P,this.i=i,l=new U.Xwb(),u=0;u<t.f;u++)e[u]==0&&(Q=t.C[u]+(t.A[u]<<16),T=U.eqb(l,h.qjb(Q)),T||(T=new a.pJ(this),U.Pwb(l,h.qjb(Q),T)),T.d=a.fJ(u,T.d),T.d.length<=T.c.length&&(T.a=T.d.length));for(D=0;D<A.f;D++)r[D]==0&&(Q=A.C[D]+(A.A[D]<<16),T=U.eqb(l,h.qjb(Q)),T||(T=new a.pJ(this),U.Pwb(l,h.qjb(Q),T)),T.c=a.fJ(D,T.c),T.c.length<=T.d.length&&(T.a=T.c.length));for(this.a=new U.Mj(),c=(y=new U.Epb(l).a.cd().Tb(),new U.Lpb(y));c.a.ed();)T=(L=c.a.fd(),L.jd()),a.nJ(T)||U.sj(this.a,T);},x(410,1,{},a.dJ),s.b=0,s.g=0,o.qab=N(410),a.eJ=function(t){t.d=B(o.Q7,H,6,0,15,1),t.c=B(o.Q7,H,6,0,15,1);},a.fJ=function(t,A){var e,r;for(r=B(o.Q7,H,6,A.length+1,15,1),e=0;e<A.length;e++)r[e]=A[e];return r[A.length]=t,r;},a.gJ=function(t,A){var e,r,P,i,D,u;for(r=t,P=0,i=r.length;P<i;++P)if(e=r[P],A.j[e]!=0)return  false;for(D=n.Bl(A,t[0]),u=1;u<t.length;u++)if(n.Bl(A,t[u])!=D)return  false;return  true;},a.hJ=function(t,A,e){var r,P,i,D,u,Q;if(Q=U.xj(t.b,A),t.d.length<=t.c.length)for(u=0;u<t.a;u++)e[t.d[u]]=t.c[Q[u]];else {for(P=t.d,i=0,D=P.length;i<D;++i)r=P[i],e[r]=-1;for(u=0;u<t.a;u++)e[t.d[Q[u]]]=t.c[u];}},a.iJ=function(t){var A,e,r;for(r=d.Math.max(t.d.length,t.c.length),e=1,A=r-t.a+1;A<=r;A++)e*=A;return e;},a.jJ=function(t,A,e,r){var P,i;if(t.d.length>t.c.length)for(P=0;P<t.a;P++)e[r+P]=t.c[P];else for(i=U.xj(t.b,A),P=0;P<t.a;P++)e[r+P]=t.c[i[P]];},a.kJ=function(t,A,e,r){var P,i;if(t.d.length<=t.c.length)for(P=0;P<t.a;P++)e[r+P]=t.d[P];else for(i=U.xj(t.b,A),P=0;P<t.a;P++)e[r+P]=t.d[i[P]];},a.lJ=function(t){var A,e;return t.b=new U.Mj(),e=B(o.Q7,H,6,t.a,15,1),A=B(o.vfb,Tt,6,d.Math.max(t.d.length,t.c.length),16,1),a.oJ(t,0,A,e),t.b.a.length;},a.mJ=function(t,A,e){var r,P;if(P=U.xj(t.b,A),t.d.length<=t.c.length){for(r=0;r<t.a;r++)if(e[t.d[r]]!=null&&e[t.d[r]][t.c[P[r]]])return  true;}else for(r=0;r<t.a;r++)if(e[t.d[P[r]]]!=null&&e[t.d[P[r]]][t.c[r]])return  true;return  false;},a.nJ=function(t){var A,e,r;if(t.a==0)return  true;if(t.d.length==1&&t.c.length==1)return (t.e.i==null||t.e.i[t.d[0]]==null||!t.e.i[t.d[0]][t.c[0]])&&(++t.e.b,t.e.f[t.d[0]]=t.e.b,t.e.d[t.c[0]]=t.e.b),true;if(a.gJ(t.d,t.e.e)||a.gJ(t.c,t.e.c)){for(A=d.Math.min(t.d.length,t.c.length),r=0;A!=0&&r<t.d.length;r++)for(e=0;e<t.c.length;e++)if(t.e.d[t.c[e]]==0&&(t.e.i==null||t.e.i[t.d[r]]==null||!t.e.i[t.d[r]][t.c[e]])){--A,++t.e.b,t.e.f[t.d[r]]=t.e.b,t.e.d[t.c[e]]=t.e.b;break;}return  true;}return  false;},a.oJ=function(t,A,e,r){var P;for(P=0;P<e.length;P++)e[P]||(e[P]=true,r[A]=P,A+1==r.length?U.sj(t.b,U.zrb(r,r.length)):a.oJ(t,A+1,e,r),e[P]=false);},a.pJ=function(t){this.e=t,a.eJ(this);},x(201,1,{201:1},a.pJ),s.a=0,o.pab=N(201),a.qJ=function(t,A){return t.b<A.b?-1:t.b>A.b?1:t.a<A.a?-1:t.a>A.a?1:0;},a.rJ=function(t,A){this.b=t,this.a=A;},x(125,1,{125:1,32:1},a.rJ),s.Sb=function(t){return a.qJ(this,t);},s.a=0,s.b=0,o.uab=N(125),a.tJ=function(t,A){var e;return e=a.xJ(t.a,A),t.a=a.wJ(t.a,e),e;},a.uJ=function(t){t.a=t.b;},a.vJ=function(){this.b=new a.zJ(),this.a=this.b;},x(409,1,{},a.vJ),o.rab=N(409),a.wJ=function(t,A){return t.a[A];},a.xJ=function(t,A){if(t.b==-1)t.a=B(o.sab,R1,170,A,0,1),t.b=A,t.c=A-1;else do t.c=t.c==0?t.b-1:t.c-1;while(t.a[t.c]&&a.yJ(t.a[t.c]));return !t.a[t.c]&&(t.a[t.c]=new a.zJ()),t.c;},a.yJ=function(t){var A,e,r,P,i;if(!t.d)if(t.b==-1)t.d=true;else {for(i=false,e=t.a,r=0,P=e.length;r<P;++r)if(A=e[r],!A)i=true;else if(!a.yJ(A))return  false;t.d=!i;}return t.d;},a.zJ=function(){this.b=-1;},x(170,1,{170:1},a.zJ),s.b=0,s.c=0,s.d=false,o.sab=N(170),a.AJ=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T,c;for(c=B(o.vfb,Tt,6,r.length,16,1),u=t,l=0,y=u.length;l<y;++l)if(i=u[l],A[i]==0){if(r[e[i]]||c[e[i]])return  false;c[e[i]]=true;}for(T=-1,D=t,Q=0,L=D.length;Q<L;++Q)if(i=D[Q],A[i]==0){if(T==-1)T=P[i];else if(P[i]!=T)return  false;}return  true;},a.BJ=function(t){var A,e,r,P,i,D,u,Q;for(t.t=new U.Mj(),r=1,i=0,e=0;e<t.u.f;e++)n.Ll(t.u,e)!=0&&!n.Am(t.u,e)&&(i=d.Math.max(i,n.Ll(t.u,e)));if(i!=0){for(P=B(o.Q7,H,6,i+1,15,1),A=0;A<t.u.f;A++)n.Ll(t.u,A)!=0&&!n.Am(t.u,A)&&(P[n.Ll(t.u,A)]=A+1);for(u=0;u<t.D.f;u++)Q=n.Ll(t.D,u),Q!=0&&Q<=i&&!n.Am(t.D,u)&&P[Q]!=0&&(D=P[Q]-1,t.I[u]=r,t.B[D]=r++,U.sj(t.t,new a.rJ(u,D)));}return t.t.a.length;},a.CJ=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c;for(y=a.FJ(t.D),u=a.FJ(t.u),D=B(o.ieb,o1,44,8,0,1),c=a.DJ(t.D,y),l=a.DJ(t.u,u),L=0;L<=7;L++)for(D[L]=new U.Ywb(new Lt.p4()),i=(r=new U.fxb(new U.lxb(new U.sqb(c[L]).a).b),new U.Bqb(r));U.apb(i.a.a);)P=(e=U.dxb(i.a),e.hd()),T=U.eqb(c[L],P),Q=U.eqb(l[L],P),Q!=null&&(A=B(o.Q7,E1,7,2,0,2),A[0]=T,A[1]=Q,U.Pwb(D[L],P,A));return D;},a.DJ=function(t,A){var e,r,P,i,D;for(i=B(o.ieb,o1,44,8,0,1),D=0;D<=7;D++)for(i[D]=new U.Ywb(new Lt.p4()),e=0;e<t.f;e++)P=A[e][D],P!=null&&(r=U.eqb(i[D],P),r=r==null?B(o.Q7,H,6,1,15,1):U.zrb(r,r.length+1),r[r.length-1]=e,U.Pwb(i[D],P,r));return i;},a.EJ=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;for(D=B(o.Q7,H,6,t.s,15,1),P=0;P<A.f;P++){for(c=A.j[P]+(A.i[P].length-A.e[P]),T=0,Q=0;Q<A.i[P].length;Q++)if(Q<A.j[P]||Q>=A.e[P]){for(b=2*e[A.i[P][Q]],i=A.k[P][Q],n.fm(A,i)==2&&(i<A.g&&n.pt(A.p,i)||++b),L=0;L<T&&!(b<D[L]);L++);for(y=T;y>L;y--)D[y]=D[y-1];D[L]=b,++T;}for(n.eh(t.b[P],P),n.ah(t.b[P],t.a,o.Kfb(e[P])),l=c;l<t.s;l++)n.ah(t.b[P],t.a+1,0);for(u=0;u<c;u++)n.ah(t.b[P],t.a+1,o.Kfb(D[u]));}for(r=A.f;r<t.b.length;r++)for(n.eh(t.b[r],r),n.ah(t.b[r],t.a,Ct(A.f+1)),u=0;u<t.s;u++)n.ah(t.b[r],t.a+1,0);},a.FJ=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(n.bw(t,7),Q=new n.uw(t.f,t.g),u=U1(o.M7,[cr,cA],[58,10],0,[t.f,8],2),r=B(o.Q7,H,6,t.f,15,1),P=B(o.vfb,Tt,6,t.f,16,1),i=B(o.vfb,Tt,6,t.g,16,1),g=0;g<t.f;g++){for(g!=0&&(U.Nrb(P,P.length),U.Nrb(i,i.length)),u[g][0]=B(o.M7,Y,6,2,15,1),u[g][0][0]=t.C[g]<<24>>24,u[g][0][1]=t.A[g]<<24>>24,r[0]=g,P[g]=true,c=0,T=1,t.u[g]|=512,k=1;k<=7&&T<t.f;k++){for(b=T,l=c;l<T;l++)for(e=r[l],L=0;L<t.j[e];L++)D=t.i[e][L],P[D]||(P[D]=true,r[b++]=D),i[t.k[e][L]]=true;if(b==T)break;for(c=T,T=b,n.Do(t,Q,i,true,null),A=0;A<Q.q;A++)Q.s[A]=0,Q.T=0,n.wn(Q,A,0);u[g][k]=w.sAb((y=n.rg(new n.Og(Q,16)),q1.Cnb(),y));}t.u[g]&=-513;}return u;},a.GJ=function(t,A){var e,r,P;for(e=0,U.Trb(t.b),r=0;r<A.length;r++)(r==0||n.dh(t.b[r],t.b[r-1])!=0)&&++e,P=t.b[r].a,A[P]=e;return e;},a.HJ=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T;for(T=new U.Oxb(new Lt.G4()),l=A,L=0,y=l.length;L<y;++L)for(Q=l[L],P=e,i=0,D=P.length;i<D;++i)r=P[i],u=B(o.Q7,H,6,2,15,1),u[0]=t.J[Q],u[1]=t.C[r],U.Pwb(T.a,u,(h.Ghb(),Fe))==null;return U.Mi(T,B(o.Q7,E1,7,0,0,2));},a.IJ=function(t,A,e){return n.$o(t.u,e)!=0&&(t.H[t.G[A]]=true),n.$o(t.D,A)!=0&&(t.A[t.w[e]]=true),new a.rJ(A,e);},a.JJ=function(t,A,e,r,P){var i,D,u,Q,l;for(i=e[r],l=false,u=0;u<A.f;u++)if(u!=r&&e[u]==i){l=true;break;}if(!l)return P;for(D=0;D<A.f;D++)(D==r||e[D]>i)&&++e[D];++P;do Q=P,a.EJ(t,A,e),P=a.GJ(t,e);while(Q!=P);return P;},a.KJ=function(t){var A,e,r,P,i,D,u;for(D=B(o.M7,cr,58,8,0,3),u=0;u<=7;u++)for(D[u]=B(o.M7,cA,10,t[u].c,0,2),r=0,i=(e=new U.fxb(new U.lxb(new U.sqb(t[u]).a).b),new U.Bqb(e));U.apb(i.a.a);)P=(A=U.dxb(i.a),A.hd()),D[u][r++]=P;return D;},a.LJ=function(t,A){var e,r,P,i,D,u,Q,l,L;for(L=0,P=t,D=0,Q=P.length;D<Q;++D)e=P[D],A[e]==0&&++L;if(L==0)return null;for(l=B(o.Q7,H,6,L,15,1),L=0,r=t,i=0,u=r.length;i<u;++i)e=r[i],A[e]==0&&(l[L++]=e);return l;},a.MJ=function(t){return t.K++==64?false:t.i?a.yJ(t.i.b)?false:(a.SJ(t),true):(t.i=new a.vJ(),true);},a.NJ=function(t){var A,e,r,P,i;for(i=d.Math.max(t.D.f,t.u.f),t.a=n.Pg(i),t.s=2,e=0;e<t.D.f;e++)t.s=d.Math.max(t.s,n.$o(t.D,e)+n.rp(t.D,e));for(r=0;r<t.u.f;r++)t.s=d.Math.max(t.s,n.$o(t.u,r)+n.rp(t.u,r));for(P=d.Math.max(2,(62+t.a+t.s*(t.a+5))/63|0),t.b=B(o.a8,Hn,101,i,0,1),A=0;A<i;A++)t.b[A]=new n.fh(P);t.F=new n.Og(t.D,1),t.v=new n.Og(t.u,1);},a.OJ=function(t){var A,e,r,P,i,D,u,Q,l;for(l=t.F.g,t.J=U.zrb(l,l.length),A=t.v.g,t.C=U.zrb(A,A.length),P=t.J,D=0,Q=P.length;D<Q;++D)e=P[D],t.o<e&&(t.o=e);for(r=t.C,i=0,u=r.length;i<u;++i)e=r[i],t.n<e&&(t.n=e);},a.PJ=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g;if(g=a.LJ(A,t.I),g==null||(b=a.LJ(e,t.B),b==null))return null;if(g.length==1&&b.length==1)return new a.rJ(g[0],b[0]);if(c=a.HJ(t,g,b),c.length==1)return new a.rJ(g[0],b[0]);for(L=a.tJ(t.i,c.length),T=-1,i=g,u=0,l=i.length;u<l;++u)if(r=i[u],t.J[r]==c[L][0]){T=r;break;}for(y=-1,P=b,D=0,Q=P.length;D<Q;++D)if(r=P[D],t.C[r]==c[L][1]){y=r;break;}return t.o=a.JJ(t,t.D,t.J,T,t.o),t.n=a.JJ(t,t.u,t.C,y,t.n),new a.rJ(T,y);},a.QJ=function(t){var A,e,r,P,i;for(r=a.RJ(t);r;){for(i=false,e=0;e<n.$o(t.D,r.b);e++)t.I[n.Zo(t.D,r.b,e)]==0&&(i=true);for(P=false,A=0;A<n.$o(t.u,r.a);A++)t.B[n.Zo(t.u,r.a,A)]==0&&(P=true);if(i&&P)break;t.I[r.b]=br,t.B[r.a]=br,r=a.RJ(t);}if(!r){for(e=0;e<t.I.length;e++)t.I[e]==br&&(t.I[e]=0);for(A=0;A<t.B.length;A++)t.B[A]==br&&(t.B[A]=0);}return r;},a.RJ=function(t){for(var A,e,r,P,i;t.t.a.length!=0;)if(r=U.Ej(t.t,0),t.I[r.b]==0&&t.B[r.a]==0)return r;for(;t.g>=0;){for(;t.g>=2&&t.c<t.j[t.g].length;){if(e=t.j[t.g][t.c],A=U.eqb(t.k[t.g],e),n.Tl(t.D,A[0][0])==6&&(r=a.PJ(t,A[0],A[1]),r))return r;++t.c;}for(;t.g>=2&&t.d<t.j[t.g].length;){if(e=t.j[t.g][t.d],A=U.eqb(t.k[t.g],e),n.Tl(t.D,A[0][0])!=6&&(r=a.PJ(t,A[0],A[1]),r))return r;++t.d;}for(;t.p&&t.g==0&&t.e<t.j[0].length;)if(e=t.j[0][t.e++],A=U.eqb(t.k[t.g],e),A[0].length==1&&A[1].length==1&&(r=a.TJ(t,A[0][0],A[1][0]),r))return r;for(;t.g==0&&t.f<t.j[t.g].length;)if(e=t.j[t.g][t.f++],A=U.eqb(t.k[t.g],e),A[0].length>=A[1].length&&a.AJ(A[0],t.I,t.G,t.H,t.J)||A[1].length>=A[0].length&&a.AJ(A[1],t.B,t.w,t.A,t.C)){for(i=0,P=0;i<A[0].length&&t.I[A[0][i]]!=0;)++i;for(;P<A[1].length&&t.B[A[1][P]]!=0;)++P;if(i<A[0].length&&P<A[1].length&&(r=a.TJ(t,A[0][i],A[1][P]),r))return r;}--t.g,t.c=0,t.d=0,t.e=0,t.f=0;}return null;},a.SJ=function(t){U.Drb(t.I,0),U.Drb(t.B,0),U.Hrb(t.H),U.Hrb(t.A),a.OJ(t),a.uJ(t.i),t.q=a.BJ(t),t.g=7,t.c=0,t.d=0,t.e=0,t.f=0;},a.TJ=function(t,A,e){return t.L!=null&&t.L[A]!=null&&t.L[A][e]?null:t.I[A]==0&&t.B[e]==0?a.IJ(t,A,e):null;},a.UJ=function(t,A,e,r,P){var i,D,u,Q,l;for(this.D=t,this.u=A,this.I=e,this.B=r,this.L=P,a.NJ(this),a.OJ(this),this.k=a.CJ(this),this.j=a.KJ(this.k),this.r=0,l=(u=new U.fxb(new U.lxb(new U.sqb(this.k[0]).a).b),new U.Bqb(u));U.apb(l.a.a);)Q=(D=U.dxb(l.a),D.hd()),i=U.eqb(this.k[0],Q),this.r+=d.Math.min(i[0].length,i[1].length);this.p=this.r==this.D.f&&this.D.f==this.u.f,this.G=B(o.Q7,H,6,this.D.f,15,1),this.w=B(o.Q7,H,6,this.u.f,15,1),this.H=B(o.vfb,Tt,6,n.hp(this.D,this.G,false,false),16,1),this.A=B(o.vfb,Tt,6,n.hp(this.u,this.w,false,false),16,1),this.q=a.BJ(this),this.g=7,this.K=0;},x(408,1,{},a.UJ),s.a=0,s.c=0,s.d=0,s.e=0,s.f=0,s.g=0,s.n=0,s.o=0,s.p=false,s.q=0,s.r=0,s.s=0,s.K=0,o.tab=N(408),a.VJ=function(t,A,e){var r,P,i,D,u;for(this.a=B(o.Q7,H,6,t.f,15,1),this.a[0]=A,e[A]=true,i=0,D=0;i<=D;){for(P=n.$o(t,this.a[i]),u=0;u<P;u++)r=n.Zo(t,this.a[i],u),d.Math.abs(t.w[r])==0&&S1(J(t.B[r],i0),0)&&!e[r]&&(this.a[++D]=r,e[r]=true);++i;}},x(405,1,{},a.VJ),o.vab=N(405),a.WJ=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E;for(n.bw(t,7),L=new n.uw(t.f,t.g),l=U1(o.M7,[b0,cr],[204,58],0,[t.f,8],2),r=B(o.Q7,H,6,t.f,15,1),P=B(o.Q7,H,6,t.f,15,1),i=B(o.vfb,Tt,6,t.f,16,1),p=0;p<t.f;p++)if(l[p]=U1(o.M7,[cr,cA],[58,10],0,[t.j[p],8],2),!(A&&t.C[p]!=6))for(Q=0;Q<t.j[p];Q++){for(y=t.i[p][Q],U.Nrb(i,i.length),v=1,k=2,t.u[y]|=512,E=0;E<8&&k<t.f;E++){if(E==0)r[0]=y,i[y]=true,r[1]=p,i[p]=true;else {for(S=k,c=v;c<k;c++)for(e=r[c],b=0;b<t.j[e];b++)u=t.i[e][b],!i[u]&&(!A||t.C[u]==6)&&(i[u]=true,r[S++]=u);if(S==k){if(!A)for(T=E;T<8;T++)l[p][Q][T]=l[p][Q][T-1];break;}v=k,k=S;}if(E==0)l[p][Q][E]=B(o.M7,Y,6,2,15,1),l[p][Q][E][0]=t.C[p]<<24>>24,l[p][Q][E][1]=t.A[p]<<24>>24;else {for(n.Co(t,L,i,true,P),n.jn(L,P[y],0),n.wn(L,P[y],0),e=0;e<t.f;e++)P[e]!=-1&&t.j[e]>n.$o(L,P[e])&&n.vn(L,P[e],Z1,true);if(A)for(D=0;D<L.g;D++)L.H[D]=1,L.T=0;l[p][Q][E]=w.sAb((g=n.rg(new n.Og(L,16)),q1.Cnb(),g));}}t.u[y]&=-513;}return l;},a.XJ=function(t,A,e){var r,P;for(r=0,P=0;P<t.j[A];P++)n.Tl(t,t.i[A][P])==e&&++r;return r;},a.YJ=function(t,A,e,r,P){var i,D,u,Q,l,L,y,T;for(T=0,y=-1,D=0;D<t.j.f;D++){L=U.xj(A.g,T),n.bw(L,1);do++y==L.f&&(y=0,L=At.pH(A,++T));while(L.K&&pt(J(L.B[y],i0),0));n.rn(L,y,e[D],e[D]<=P);}for(l=0,Q=-1,i=0;i<t.d.f;i++){u=U.xj(A.f,l),n.bw(u,1);do++Q==u.f&&(Q=0,u=At.oH(A,++l));while(u.K&&pt(J(u.B[Q],i0),0));n.rn(u,Q,r[i],r[i]<=P);}},a.ZJ=function(t,A,e,r,P){var i,D,u,Q,l;for(l=-1,D=0;D<n.$o(t.j,e);D++)if(n.Zo(t.j,e,D)==A){l=D;break;}for(u=-1,i=0;i<n.$o(t.d,P);i++)if(n.Zo(t.d,P,i)==r){u=i;break;}for(Q=0;Q<8;Q++)if(Lt.o4(t.k[e][l][Q],t.e[P][u][Q])!=0)return Q<<8;return B1;},a.$J=function(t,A){return A<t.g&&(n.ut(t.p,A)||t.H[A]==8)?0:t.H[A]&127;},a._J=function(t,A,e){var r,P;for(P=0;P<t.j[e];P++)if(r=t.i[e][P],r!=A)return r;return  -1;},a.aK=function(t,A){var e,r;for(e=t.o[A],r=0;r<t.j[A];r++)n.Hm(t,t.i[A][r])&&++e;return e;},a.bK=function(t,A,e,r,P){var i,D,u,Q,l;for(l=-1,D=0;D<n.$o(t.j,e);D++)if(n.Zo(t.j,e,D)==A){l=D;break;}for(u=-1,i=0;i<n.$o(t.d,P);i++)if(n.Zo(t.d,P,i)==r){u=i;break;}for(Q=2;Q<8;Q++)if(t.p[e][l][Q]==null||Lt.o4(t.p[e][l][Q],t.i[P][u][Q])!=0)return Q-2<<8;return 1536;},a.cK=function(t){var A,e,r,P,i,D,u,Q;for(u=B(o.Q7,H,6,t.f,15,1),Q=(n.bw(t,7),t.p),i=0;i<d.Math.min(32,Q.j.a.length);i++)for(D=U.xj(Q.j,i),e=D,r=0,P=e.length;r<P;++r)A=e[r],u[A]|=1<<i;return u;},a.dK=function(t,A,e,r){return (r[t]&r[A]&~r[e])!=0;},a.eK=function(t,A,e,r,P,i){var D,u,Q,l,L,y,T,c;for(t.j=A,t.d=e,t.n=B(o.Q7,H,6,r.length,15,1),t.f=B(o.Q7,H,6,P.length,15,1),t.k=a.WJ(t.j,false),t.p=a.WJ(t.j,true),t.e=a.WJ(t.d,false),t.i=a.WJ(t.d,true),t.o=a.cK(t.j),t.g=a.cK(t.d),t.q=IP,y=new a.UJ(A,e,t.n,t.f,i),t.a=0;a.MJ(y);){for(++t.a,t.c=y.q,l=y.r,L=a.QJ(y);L;)a.fK(t,L),L=a.QJ(y);if(t.b=t.c,t.c<l?(D=new a.dJ(t.j,t.d,t.n,t.f,t.c,i),T=a.cJ(D),t.c+=D.b-D.g):(c=new a.bJ(t.j,t.d),T=a.aJ(c,a.VI(c,t.n,t.f))),t.q<T){for(t.q=T,Q=0;Q<r.length;Q++)r[Q]=t.n[Q];for(u=0;u<P.length;u++)P[u]=t.f[u];}}},a.fK=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K;for(l=B(o.Q7,H,6,t.j.f,15,1),v=B(o.Q7,H,6,t.d.f,15,1),L=B(o.Q7,H,6,t.j.f,15,1),p=B(o.Q7,H,6,t.d.f,15,1),l[0]=A.b,L[A.b]=A.b,v[0]=A.a,p[A.a]=A.a,t.n[A.b]==0&&(++t.c,t.n[A.b]=t.c,t.f[A.a]=t.c),u=0,y=0;u<=y;){for(V=l[u],E=v[u],k=U1(o.Q7,[E1,H],[7,6],15,[n.$o(t.j,V),n.$o(t.d,E)],2),c=0;c<n.$o(t.j,V);c++)if(q=n.Zo(t.j,V,c),t.n[q]==0)for(i=a.$J(t.j,n._o(t.j,V,c)),g=0;g<n.$o(t.d,E);g++)S=n.Zo(t.d,E,g),t.f[S]==0&&n.Tl(t.j,q)==n.Tl(t.d,S)&&(D=n._o(t.d,E,g),K=a.bK(t,V,q,E,S),i==a.$J(t.d,D)||K!=0?a.iK(t,V,q,E,S)&&(Q=a.ZJ(t,V,q,E,S),G=d.Math.max(K,Q),G!=0&&(b=a.gK(t,L[V],V,q,p[E],E,S),a.jK(t,V,q,E,S,G,b)&&(k[c][g]=G,b&&(k[c][g]+=64),a.dK(L[V],V,q,t.o)==a.dK(p[E],E,S,t.g)&&(k[c][g]+=128),a.aK(t.j,q)==a.aK(t.d,S)&&(k[c][g]+=1)))):a.ZJ(t,V,q,E,S)==B1&&(k[c][g]=768));for(;;){for(e=0,P=-1,r=-1,T=0;T<k.length;T++)if(q=n.Zo(t.j,V,T),t.n[q]==0)for(g=0;g<k[T].length;g++)S=n.Zo(t.d,E,g),t.f[S]==0&&e<k[T][g]&&(e=k[T][g],P=q,r=S);if(e==0)break;++y,l[y]=P,L[P]=l[u],v[y]=r,p[r]=v[u],++t.c,t.n[P]=t.c,t.f[r]=t.c;}++u;}return y+1;},a.gK=function(t,A,e,r,P,i,D){var u,Q,l,L,y;if(n.$o(t.j,e)==3&&(n.Nl(t.j,e)==1||n.Nl(t.j,e)==2)&&n.$o(t.d,i)==3&&(n.Nl(t.d,i)==1||n.Nl(t.d,i)==2)){for(y=A>r,l=-1,Q=0;Q<n.$o(t.j,e);Q++)if(l=n.Zo(t.j,e,Q),l!=A&&l!=r){(l>r&&l<A||l<r&&l>A)&&(y=!y);break;}for(L=P>D,u=0;u<n.$o(t.d,i);u++)if(l=n.Zo(t.d,i,u),l!=P&&l!=D){(l>D&&l<P||l<D&&l>P)&&(L=!L);break;}return y==L==(n.Nl(t.j,e)==n.Nl(t.d,i));}return  true;},a.hK=function(t,A){var e,r,P,i;for(t.j=new n.tw(),r=0;r<A.g.a.length;r++)i=U.xj(A.g,r),n.bw(i,1),n.Zk(t.j,i,i.f,i.g,!i.K);for(t.d=new n.tw(),e=0;e<A.f.a.length;e++)P=U.xj(A.f,e),n.bw(P,1),n.Zk(t.d,P,P.f,P.g,!P.K);n.lq(t.j),n.lq(t.d);},a.iK=function(t,A,e,r,P){var i,D;return !(n.Tl(t.j,e)==6&&d.Math.abs(n.$o(t.j,e)-n.$o(t.d,P))>1||n.Tl(t.j,e)==8&&n.$o(t.j,e)==2&&a.XJ(t.j,A,8)>a.XJ(t.d,r,8)||n.Tl(t.d,P)==8&&n.$o(t.d,P)==2&&a.XJ(t.j,A,8)<a.XJ(t.d,r,8)||n.Tl(t.j,e)==8&&n.$o(t.j,e)==2&&n.$o(t.d,P)==2&&(D=n.Zo(t.j,e,0)==A?0:1,i=n.Zo(t.d,P,0)==r?0:1,Lt.o4(t.p[e][D][3],t.i[P][i][3])!=0)||n.Tl(t.j,e)==7&&a.XJ(t.j,A,7)<a.XJ(t.d,r,7)||n.Tl(t.d,P)==7&&a.XJ(t.j,A,7)>a.XJ(t.d,r,7)||n.Yo(t.j,n.Xo(t.j,A,e))==3^n.Yo(t.d,n.Xo(t.d,r,P))==3);},a.jK=function(t,A,e,r,P,i,D){return !(n.Tl(t.j,e)==8&&n.Qo(t.j,e)==0&&a.nK(t.j,A)&&a.nK(t.d,r)&&i!=B1||n.Tl(t.j,e)==8&&n.$o(t.j,e)==2&&a.nK(t.j,a._J(t.j,A,e))&&i<768||n.Tl(t.j,A)==8&&(a.nK(t.j,e)||a.nK(t.d,P))&&i<512||!D&&(n.Tl(t.j,e)!=6||!a.lK(t.j,A))||n.Tl(t.j,e)==8&&n.Tl(t.j,A)!=6&&i!=B1||n.Tl(t.j,A)==5&&n.Tl(t.j,e)==6&&i<768);},a.kK=function(){},a.lK=function(t,A){var e;for(e=0;e<t.j[A];e++)if(n.Tl(t,t.i[A][e])!=6)return  true;return  false;},a.mK=function(t,A,e){var r;for(r=0;r<t.j[A];r++)if(n.Tl(t,t.i[A][r])!=6&&t.i[A][r]!=e)return  true;return  false;},a.nK=function(t,A){var e;for(e=0;e<t.j[A];e++)if(t.n[A][e]==2&&n.Tl(t,t.i[A][e])>6)return  true;return  false;},a.oK=function(t,A,e){var r;for(r=0;r<t.j[A];r++)if(t.n[A][r]==2&&n.Tl(t,t.i[A][r])>6&&t.i[A][r]!=e)return  true;return  false;},x(404,1,{},a.kK),s.a=0,s.b=0,s.c=0,s.q=0,o.wab=N(404),C.sK=function(){C.sK=f1,C.qK=j(F(o.Ocb,1),yt,2,6,["","r0","r","r3","r3-4","r3-5","r3-6","r3-7","r4","r4-5","r4-6","r4-7","r>3","r5","r5-6","r5-7","r>4","r6","r6-7","r>5","r7","r>6","r>7"]),C.pK=j(F(o.Ocb,1),yt,2,6,[wA,kD,"is in any ring","3 members","3-4 members","3-5 members","3-6 members","3-7 members","4 members","4-5 members","4-6 members","4-7 members","> 3 members","5 members","5-6 members","5-7 members","> 4 members","6 members","6-7 members","> 5 members","7 members","> 6 members","> 7 members"]),C.rK=j(F(o.R7,1),e0,6,14,[0,lA,K8,qA,25769803776,60129542144,128849018880,266287972352,VA,51539607552,120259084288,257698037760,532575944704,IA,103079215104,240518168576,515396075520,OA,206158430208,481036337152,xA,412316860416,un]);},C.tK=function(t,A,e,r){var P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q;for(t.A=A,t.a=e,y=Mt(d.Math.round((m1.sO==-1&&(m1.sO=m1.uO()),m1.sO*8))),T=j(F(o.Q7,1),H,6,15,[y,y/2|0,y*3/2|0,y/2|0,y/2|0,y/2|0,y/2|0,y/2|0,y/2|0,y/2|0,y/2|0,y/2|0,y*3/2|0,y/4|0,y/2|0,y/2|0,y/4|0]),c=j(F(o.Q7,1),H,6,15,[y*3/2|0,y/2|0]),b=j(F(o.Q7,1),H,6,15,[y,-2,y,-2,y]),q=B(o.Q7,H,6,1+2*T.length+(r?2*c.length:0),15,1),k=0,D=T,Q=0,L=D.length;Q<L;++Q)P=D[Q],q[k++]=P,q[k++]=-2;if(r)for(i=c,u=0,l=i.length;u<l;++u)P=i[u],q[k++]=P,q[k++]=-2;for(q[k++]=y,R.e3(t.v,b,q),t.b=R.$2(t.v,"Any atomic number"),R.A2(t.b,t),R.Z2(t.v,t.b,1,1,3,1),t.w=R.a3(t.v,OP),t.D=R.b3(t.v,16,1),R.Y2(t.v,t.w,1,3),R.Y2(t.v,t.D,3,3),t.g=R._2(t.v),R.Q2(t.g,wA),R.Q2(t.g,"is aromatic"),R.Q2(t.g,"is hetero-aromatic"),R.Q2(t.g,"is not aromatic"),R.Y2(t.v,R.a3(t.v,"Aromaticity:"),1,5),R.Y2(t.v,t.g,3,5),t.s=R._2(t.v),R.Q2(t.s,wA),R.Q2(t.s,"0 (not in a ring)"),R.Q2(t.s,"0 or 2 (0 or 1 ring)"),R.Q2(t.s,">=2 (any ring count)"),R.Q2(t.s,"2 (in 1 ring)"),R.Q2(t.s,"3 (bridge head; 2 rings)"),R.Q2(t.s,">3 (in more than 2 rings)"),R.Y2(t.v,R.a3(t.v,"Ring bonds:"),1,7),R.Y2(t.v,t.s,3,7),t.t=R._2(t.v),R.Q2(t.t,wA),R.Q2(t.t,"3 members"),R.Q2(t.t,"4 members"),R.Q2(t.t,"5 members"),R.Q2(t.t,"6 members"),R.Q2(t.t,"7 members"),R.Y2(t.v,R.a3(t.v,"Smallest ring size:"),1,9),R.Y2(t.v,t.t,3,9),t.r=R._2(t.v),S=C.pK,p=0,E=S.length;p<E;++p)v=S[p],R.Q2(t.r,v);for(R.Y2(t.v,R.a3(t.v,"Any ring size:"),1,11),R.Y2(t.v,t.r,3,11),t.i=R._2(t.v),R.Q2(t.i,wA),R.Q2(t.i,"not charged"),R.Q2(t.i,"has negative charge"),R.Q2(t.i,"has positive charge"),R.Y2(t.v,R.a3(t.v,"Charge:"),1,13),R.Y2(t.v,t.i,3,13),t.n=R._2(t.v),R.Q2(t.n,wA),R.Q2(t.n,wr),R.Q2(t.n,gr),R.Q2(t.n,"exactly 3"),R.Q2(t.n,xP),R.Q2(t.n,dD),R.Q2(t.n,Br),R.Q2(t.n,NP),R.Q2(t.n,GP),R.Q2(t.n,mD),R.Q2(t.n,zP),R.Q2(t.n,"1,2, or 3"),R.Q2(t.n,"2 or 3"),R.Y2(t.v,R.a3(t.v,"Non-H neighbours:"),1,15),R.Y2(t.v,t.n,3,15),t.j=R._2(t.v),R.Q2(t.j,wA),R.Q2(t.j,"exactly 0"),R.Q2(t.j,wr),R.Q2(t.j,gr),R.Q2(t.j,"exactly 3"),R.Q2(t.j,vD),R.Q2(t.j,xP),R.Q2(t.j,dD),R.Q2(t.j,Br),R.Q2(t.j,NP),R.Q2(t.j,GP),R.Q2(t.j,mD),R.Q2(t.j,zP),R.Q2(t.j,"1,2, or 3"),R.Q2(t.j,"2 or 3"),R.Y2(t.v,R.a3(t.v,"Electroneg. neighbours:"),1,17),R.Y2(t.v,t.j,3,17),t.k=R._2(t.v),R.Q2(t.k,wA),R.Q2(t.k,"none"),R.Q2(t.k,wr),R.Q2(t.k,gr),R.Q2(t.k,Br),R.Q2(t.k,NP),R.Q2(t.k,GP),R.Q2(t.k,vD),R.Q2(t.k,xP),R.Q2(t.k,zP),R.Y2(t.v,R.a3(t.v,"Hydrogen count:"),1,19),R.Y2(t.v,t.k,3,19),t.p=R._2(t.v),R.Q2(t.p,wA),R.Q2(t.p,"none"),R.Q2(t.p,wr),R.Q2(t.p,gr),R.Q2(t.p,Br),R.Y2(t.v,R.a3(t.v,"Pi-electron count:"),1,21),R.Y2(t.v,t.p,3,21),t.o=R._2(t.v),R.Q2(t.o,wA),g=1;g<15;g++)R.Q2(t.o,""+(g-7));R.Y2(t.v,R.a3(t.v,"Oxidation state:"),1,23),R.Y2(t.v,t.o,3,23),t.c=R.$2(t.v,"prohibit further substitution"),R.A2(t.c,t),R.Z2(t.v,t.c,1,25,3,25),t.f=R.$2(t.v,"require further substitution"),R.A2(t.f,t),R.Z2(t.v,t.f,1,27,3,27),t.u=R._2(t.v),R.Q2(t.u,wA),R.Q2(t.u,"is a stereo center"),R.Q2(t.u,"is not a stereo center"),R.Y2(t.v,R.a3(t.v,"Stereo center:"),1,29),R.Y2(t.v,t.u,3,29),t.e=R.$2(t.v,"match stereo center"),R.Z2(t.v,t.e,1,31,3,31),t.d=R.$2(t.v,"is part of exclude group"),R.Z2(t.v,t.d,1,33,3,33),r&&(R.Z2(t.v,R.a3(t.v,"Stereo center hint for product:"),1,35,3,35),t.q=R._2(t.v),R.Q2(t.q,"Copy from generic product"),R.Q2(t.q,"Keep reactant configuration"),R.Q2(t.q,"Invert reactant configuration"),R.Q2(t.q,"Racemise configuration"),R.Z2(t.v,t.q,1,37,3,37)),n.bw(t.A,31),C.xK(t);},C.uK=function(t){var A,e,r,P,i,D,u,Q,l;for(u=null,Q=h.skb(R.Y3(t.D));w.SAb(Q).length!=0;){if(e=h.hkb(Q,X1(44)),e==-1?(D=h.skb(Q),Q=""):(D=h.skb((w.MAb(0,e,w.SAb(Q).length),w.SAb(Q).substr(0,e))),Q=h.skb((w.NAb(e+1,w.SAb(Q).length+1),w.SAb(Q).substr(e+1)))),h.akb(D,"X")){w.SAb(Q).length!=0&&(Q=","),Q=(w.GAb(Q),Q+(w.GAb("F,Cl,Br,I"),"F,Cl,Br,I"));continue;}if(A=(n.Rk(),n.po(D,321)),A!=0)if(A==1)R.g3(t.v,"'H' cannot be part of an atom list and is removed.");else if(u==null)u=B(o.Q7,H,6,1,15,1),u[0]=A;else {for(r=false,i=0;i<u.length;i++)if(A==u[i]){r=true;break;}if(!r){for(l=B(o.Q7,H,6,u.length+1,15,1),P=0;P<u.length;P++)l[P]=u[P];l[u.length]=A,u=l;}}}return u!=null&&w.lAb(u,w.SAb(T0(U.lsb.prototype.ld,U.lsb,[]))),u;},C.vK=function(t,A){A.j==0?(C.yK(t),t.B=true,R.c3(t.v)):A.j==1?R.c3(t.v):F1(A.k)===F1(t.b)?A.a==1?(R.Z3(t.D,""),R.R3(t.w,OP)):(R.Z3(t.D,n.Il(t.A,t.a)),R.R3(t.w,FD)):F1(A.k)===F1(t.c)?(R.I2(t.f,false),R.V2(t.n,0),R.V2(t.j,0)):F1(A.k)===F1(t.f)&&R.I2(t.c,false);},C.wK=function(t,A){C.JM(A,t.B);},C.xK=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k;for(L=n.Ol(t.A,t.a),pt(J(L,1),0)?(R.I2(t.b,true),R.R3(t.w,OP)):R.R3(t.w,FD),R.Z3(t.D,n.Jl(t.A,t.a)==null?"":n.Kl(t.A,t.a)),A=J(L,Xe),pt(J(A,ee),0)?R.V2(t.g,2):nt(A,2)==0?R.V2(t.g,1):nt(A,4)==0?R.V2(t.g,3):R.V2(t.g,0),c=J(L,120),nt(c,112)==0?R.V2(t.s,1):nt(c,96)==0?R.V2(t.s,2):nt(c,8)==0?R.V2(t.s,3):nt(c,104)==0?R.V2(t.s,4):nt(c,88)==0?R.V2(t.s,5):nt(c,56)==0?R.V2(t.s,6):R.V2(t.s,0),g=Jt(A0(J(L,vA),22)),R.V2(t.t,g==0?0:g-2),T=J(L,EA),u=-1,D=0;D<C.rK.length;D++)if(S1(T,C.rK[D])){u=D;break;}u!=-1?R.V2(t.r,u):(r=new h.Mkb("Custom:"),pt(J(T,lA),0)&&(r.a+=" 0"),pt(J(T,qA),0)&&(r.a+=" 3"),pt(J(T,VA),0)&&(r.a+=" 4"),pt(J(T,IA),0)&&(r.a+=" 5"),pt(J(T,OA),0)&&(r.a+=" 6"),pt(J(T,xA),0)&&(r.a+=" 7"),pt(J(T,un),0)&&(r.a+=" >=8"),t.C=T,R.Q2(t.r,r.a),R.V2(t.r,C.rK.length)),Q=J(L,aA),nt(Q,Pe)==0?R.V2(t.n,1):nt(Q,Mn)==0?R.V2(t.n,2):nt(Q,Rn)==0?R.V2(t.n,3):nt(Q,ie)==0?R.V2(t.n,4):nt(Q,j0)==0?R.V2(t.n,5):nt(Q,r0)==0?R.V2(t.n,6):nt(Q,tn)==0?R.V2(t.n,7):nt(Q,iA)==0?R.V2(t.n,8):nt(Q,ln)==0?R.V2(t.n,9):nt(Q,3276800)==0?R.V2(t.n,10):nt(Q,2228224)==0?R.V2(t.n,11):nt(Q,2490368)==0?R.V2(t.n,12):R.V2(t.n,0),P=J(L,Ln),nt(P,Oe)==0?R.V2(t.j,1):nt(P,xe)==0?R.V2(t.j,2):nt(P,Ne)==0?R.V2(t.j,3):nt(P,Ge)==0?R.V2(t.j,4):nt(P,tP)==0?R.V2(t.j,5):nt(P,AP)==0?R.V2(t.j,6):nt(P,fe)==0?R.V2(t.j,7):nt(P,Cn)==0?R.V2(t.j,8):nt(P,ze)==0?R.V2(t.j,9):nt(P,Je)==0?R.V2(t.j,10):nt(P,De)==0?R.V2(t.j,11):nt(P,nP)==0?R.V2(t.j,12):nt(P,eP)==0?R.V2(t.j,13):nt(P,rP)==0?R.V2(t.j,14):R.V2(t.j,0),e=J(L,mA),nt(e,re)==0?R.V2(t.i,1):nt(e,on)==0?R.V2(t.i,2):nt(e,Qn)==0?R.V2(t.i,3):R.V2(t.i,0),i=J(L,X0),nt(i,1792)==0?R.V2(t.k,1):nt(i,1664)==0?R.V2(t.k,2):nt(i,1408)==0?R.V2(t.k,3):nt(i,128)==0?R.V2(t.k,4):nt(i,384)==0?R.V2(t.k,5):nt(i,896)==0?R.V2(t.k,6):nt(i,1536)==0?R.V2(t.k,7):nt(i,f0)==0?R.V2(t.k,8):nt(i,1152)==0?R.V2(t.k,9):R.V2(t.k,0),l=J(L,$A),nt(l,98304)==0?R.V2(t.p,1):nt(l,81920)==0?R.V2(t.p,2):nt(l,49152)==0?R.V2(t.p,3):nt(l,N0)==0?R.V2(t.p,4):R.V2(t.p,0),y=A0(J(L,uP),47),R.V2(t.o,Jt(y)),pt(J(L,B1),0)&&R.I2(t.c,true),pt(J(L,Z1),0)&&R.I2(t.f,true),k=J(L,DP),S1(k,Ae)?R.V2(t.u,1):S1(k,ne)?R.V2(t.u,2):R.V2(t.u,0),pt(J(L,L0),0)&&R.I2(t.e,true),pt(J(L,i0),0)&&R.I2(t.d,true),t.q&&(b=J(L,hA),nt(b,I0)==0?R.V2(t.q,1):nt(b,ke)==0?R.V2(t.q,2):nt(b,hA)==0?R.V2(t.q,3):R.V2(t.q,0));},C.yK=function(t){var A,e;if(e=C.uK(t),n.Pm(t.A,t.a))for(A=0;A<t.A.q;A++)n.Pm(t.A,A)&&C.zK(t,A,e);else C.zK(t,t.a,e);n.Cq(t.A);},C.zK=function(t,A,e){var r,P,i,D,u,Q;for(i=0,R.H2(t.b)?(i=Pt(i,1),n.qn(t.A,A,e,true)):n.qn(t.A,A,e,false),R.R2(t.g)==2?n.aq(t.A,A)||(i=Pt(i,sP)):n.Tp(t.A,A)||(R.R2(t.g)==1?i=Pt(i,2):R.R2(t.g)==3&&(i=Pt(i,4))),u=0,P=0;P<n.$o(t.A,A);P++)pt(J(n.Ol(t.A,n.Zo(t.A,A,P)),i0),0)&&n.gq(t.A,n._o(t.A,A,P))&&++u;switch(R.R2(t.s)){case 1:u==0&&(i=Pt(i,112));break;case 2:u<=2&&(i=Pt(i,96));break;case 3:i=Pt(i,8);break;case 4:u<3&&(i=Pt(i,104));break;case 5:u<4&&(i=Pt(i,88));break;case 6:i=Pt(i,56);}switch(R.R2(t.t)!=0&&(i=Pt(i,h1(Ct(R.R2(t.t)+2),22))),Q=R.R2(t.r),Q==C.rK.length?i=Pt(i,t.C):Q!=0&&(i=Pt(i,C.rK[Q])),R.R2(t.i)){case 1:i=Pt(i,re);break;case 2:i=Pt(i,on);break;case 3:i=Pt(i,Qn);}switch(D=n.up(t.A,A),R.R2(t.n)){case 1:D==1?i=Pt(i,B1):D<1&&(i=Pt(i,Pe));break;case 2:D==2?i=Pt(i,B1):D<2&&(i=Pt(i,Mn));break;case 3:D==3?i=Pt(i,B1):D<3&&(i=Pt(i,Rn));break;case 4:D==2?i=Pt(i,B1):D<2&&(i=Pt(i,ie));break;case 5:D==3?i=Pt(i,B1):D<3&&(i=Pt(i,j0));break;case 6:D==0&&(i=Pt(i,Z1));break;case 7:D==1?i=Pt(i,Z1):D<1&&(i=Pt(i,tn));break;case 8:D==2?i=Pt(i,Z1):D<2&&(i=Pt(i,iA));break;case 9:D==3?i=Pt(i,Z1):D<3&&(i=Pt(i,ln));break;case 10:D==0?i=Pt(i,3276800):i=Pt(i,ie);break;case 11:D==0?i=Pt(i,2228224):i=Pt(i,j0);break;case 12:D<=1?i=Pt(i,1099513856e3):D<=3&&(i=Pt(i,j0));}switch(r=n.Po(t.A,A),R.R2(t.j)){case 1:r==0&&(i=Pt(i,Oe));break;case 2:r<=1&&(i=Pt(i,xe));break;case 3:r<=2&&(i=Pt(i,Ne));break;case 4:r<=3&&(i=Pt(i,Ge));break;case 5:r<2&&(i=Pt(i,tP));break;case 6:r<3&&(i=Pt(i,AP));break;case 7:r<4&&(i=Pt(i,fe));break;case 8:r==0&&(i=Pt(i,Cn));break;case 9:r<2&&(i=Pt(i,ze));break;case 10:r<3&&(i=Pt(i,Je));break;case 11:r<4&&(i=Pt(i,De));break;case 12:r<2&&(i=Pt(i,nP));break;case 13:r<3&&(i=Pt(i,eP));break;case 14:r<3&&(i=Pt(i,rP));}switch(R.R2(t.k)){case 1:i=Pt(i,1792);break;case 2:i=Pt(i,1664);break;case 3:i=Pt(i,1408);break;case 4:i=Pt(i,128);break;case 5:i=Pt(i,384);break;case 6:i=Pt(i,896);break;case 7:i=Pt(i,1536);break;case 8:i=Pt(i,f0);break;case 9:i=Pt(i,1152);}switch(R.R2(t.p)){case 1:i=Pt(i,98304);break;case 2:i=Pt(i,81920);break;case 3:i=Pt(i,49152);break;case 4:i=Pt(i,N0);}switch(R.R2(t.o)!=0&&(i=Pt(i,h1(Ct(R.R2(t.o)),47))),R.H2(t.c)&&(n.jp(t.A,A)>0||n.Bl(t.A,A)==0&&(n.Tl(t.A,A)==5||n.bq(t.A,A)||n.Xp(t.A,A)))&&(i=Pt(i,B1)),R.H2(t.f)&&(n.jp(t.A,A)>0||n.Bl(t.A,A)==0&&(n.Tl(t.A,A)==5||n.bq(t.A,A)||n.Xp(t.A,A)))&&(i=Pt(i,Z1)),R.R2(t.u)){case 1:i=Pt(i,Ae);break;case 2:i=Pt(i,ne);}if(R.H2(t.e)&&(i=Pt(i,L0)),R.H2(t.d)&&(i=Pt(i,i0)),t.q)switch(R.R2(t.q)){case 1:i=Pt(i,I0);break;case 2:i=Pt(i,ke);break;case 3:i=Pt(i,hA);}n.vn(t.A,A,-1,false),n.vn(t.A,A,i,true);},C.AK=function(t,A){t.B=false,R.f3(t.v,new C.DK(t,A));},C.BK=function(t,A,e,r){C.sK(),this.v=W._1(t,(A.u[e]&512)!=0?"Atom Query Features (Multiple)":"Atom Query Features",this),C.tK(this,A,e,r);},x(292,1,Tn,C.BK),s.mc=function(t){C.vK(this,t);},s.a=0,s.B=false,s.C=0,o.yab=N(292),C.DK=function(t,A){this.a=t,this.b=A;},x(380,1,{},C.DK),s.nc=function(){C.wK(this.a,this.b);},o.xab=N(380),C.FK=function(t,A,e){var r,P,i,D;for(t.u=A,t.a=e,r=Mt(d.Math.round((m1.sO==-1&&(m1.sO=m1.uO()),m1.sO*8))),P=j(F(o.Q7,1),H,6,15,[r,-1,-2,-2,r]),D=j(F(o.Q7,1),H,6,15,[r,-2,r,-2,-2,-2,-2,-2,-2,-2,r,-2,r,-2,r,-2,r,-2,2*r,-2,r/2|0,-2,r/2|0,-2,2*r]),R.e3(t.s,P,D),R.Z2(t.s,R.a3(t.s,"Desired Bond type(s):"),1,1,3,1),t.k=R.$2(t.s,"Single"),R.Z2(t.s,t.k,1,3,3,3),t.c=R.$2(t.s,"Double"),R.Z2(t.s,t.c,1,4,3,4),t.n=R.$2(t.s,"Triple"),R.Z2(t.s,t.n,1,5,3,5),t.i=R.$2(t.s,"Quadruple"),R.Z2(t.s,t.i,1,6,3,6),t.j=R.$2(t.s,"Quintuple"),R.Z2(t.s,t.j,1,7,3,7),t.b=R.$2(t.s,"Delocalized"),R.Z2(t.s,t.b,1,8,3,8),t.g=R.$2(t.s,"Coordinate (0-order)"),R.Z2(t.s,t.g,1,9,3,9),t.q=R._2(t.s),R.Q2(t.q,"any ring state"),R.Q2(t.q,kD),R.Q2(t.q,"is any ring bond"),R.Q2(t.q,"is non-aromatic ring bond"),R.Q2(t.q,"is aromatic bond"),R.Q2(t.q,"is any non-aromatic bond"),R.A2(t.q,t),R.Z2(t.s,t.q,1,11,3,11),t.r=R._2(t.s),R.Q2(t.r,"any ring size"),R.Q2(t.r,"is in 3-membered ring"),R.Q2(t.r,"is in 4-membered ring"),R.Q2(t.r,"is in 5-membered ring"),R.Q2(t.r,"is in 6-membered ring"),R.Q2(t.r,"is in 7-membered ring"),R.Q2(t.r,"smallest ring 8 to 11"),R.Q2(t.r,"smallest ring >= 12"),R.Z2(t.s,t.r,1,13,3,13),t.e=R.$2(t.s,"Match formal bond order"),R.I2(t.e,(A.G[e]&sn)!=0),R.A2(t.e,t),R.Z2(t.s,t.e,1,15,3,15),t.f=R.$2(t.s,"Match Stereo Configuration"),R.I2(t.f,(A.G[e]&TA)!=0),R.A2(t.f,t),R.Z2(t.s,t.f,1,17,3,17),t.d=R.$2(t.s,"Is atom bridge between"),R.A2(t.d,t),R.Z2(t.s,t.d,1,19,3,19),t.p=R._2(t.s),i=0;i<16;i++)R.Q2(t.p,""+i);R.Y2(t.s,t.p,2,21),R.Y2(t.s,R.a3(t.s," and"),3,21),R.A2(t.p,t),t.o=R._2(t.s),C.KK(t,0),R.Y2(t.s,t.o,2,23),R.Y2(t.s,R.a3(t.s," atoms"),3,23),n.bw(t.u,7),C.LK(t);},C.GK=function(t){var A;A=R.H2(t.d),R.C2(t.k,!A),R.C2(t.c,!A),R.C2(t.n,!A),R.C2(t.i,!A),R.C2(t.j,!A),R.C2(t.b,!A),R.C2(t.g,!A),R.C2(t.f,!A&&n.fm(t.u,t.a)==2&&n.gm(t.u,t.a)!=0&&n.gm(t.u,t.a)!=3),R.C2(t.q,!A),R.C2(t.r,!A&&R.R2(t.q)!=1),R.C2(t.p,A),R.C2(t.o,A);},C.HK=function(t,A){var e,r,P;A.j==1?R.c3(t.s):A.j==0?(C.MK(t),t.v=true,R.c3(t.s)):F1(A.k)===F1(t.d)||F1(A.k)===F1(t.q)?C.GK(t):F1(A.k)===F1(t.p)&&(P=R.R2(t.p),t.t!=P&&(r=t.t+R.R2(t.o),e=C.KK(t,P),r<P?R.V2(t.o,0):r<P+e?R.V2(t.o,r-P):R.V2(t.o,e-1),t.t=P));},C.IK=function(t,A){return n.Pm(t.u,n.Zl(t.u,0,A))&&n.Pm(t.u,n.Zl(t.u,1,A));},C.JK=function(t,A){C.LM(A,t.v);},C.KK=function(t,A){var e;for(R.T2(t.o),e=0;e<16;e++)R.Q2(t.o,""+(A+e));return 16;},C.LK=function(t){var A,e,r,P,i,D,u;i=n.hm(t.u,t.a),r=n.im(t.u,t.a)==8||n.Yp(t.u,t.a)?6:n.fm(t.u,t.a),((i&1)!=0||r==1)&&R.I2(t.k,true),((i&2)!=0||r==2)&&R.I2(t.c,true),((i&4)!=0||r==3)&&R.I2(t.n,true),((i&32)!=0||r==4)&&R.I2(t.i,true),((i&64)!=0||r==5)&&R.I2(t.j,true),((i&8)!=0||r==6)&&R.I2(t.b,true),((i&16)!=0||r==0)&&R.I2(t.g,true),(i&sn)!=0&&R.I2(t.e,true),(i&TA)!=0&&R.I2(t.f,true),u=i&384,A=i&Qe,u==128?R.V2(t.q,1):A==j0?R.V2(t.q,4):u==256?R.V2(t.q,A==0?2:3):R.V2(t.q,A==0?0:5),D=(i&iA)>>17,R.V2(t.r,D==0?0:D<=2?D+5:D-2),(i&sA)!=0?(R.I2(t.d,true),P=(i&7680)>>9,e=(i&$e)>>13,R.V2(t.p,P),C.KK(t,P),R.V2(t.o,e)):R.V2(t.o,0),C.GK(t);},C.MK=function(t){var A;if(C.IK(t,t.a))for(A=0;A<t.u.r;A++)n.Pm(t.u,n.Zl(t.u,0,A))&&n.Pm(t.u,n.Zl(t.u,1,A))&&C.NK(t,A);else C.NK(t,t.a);n.Dq(t.u);},C.NK=function(t,A){var e,r,P,i,D;i=0,R.H2(t.d)?(P=R.R2(t.p),e=R.R2(t.o),i|=P<<9,i|=e<<13,i&=-128,n.Mn(t.u,A,1)):(R.H2(t.k)&&(i|=1),R.H2(t.c)&&(i|=2),R.H2(t.n)&&(i|=4),R.H2(t.i)&&(i|=32),R.H2(t.j)&&(i|=64),R.H2(t.b)&&(i|=8),R.H2(t.g)&&(i|=16),R.R2(t.q)!=0&&(R.R2(t.q)==1?n.gq(t.u,A)||(i|=128):R.R2(t.q)==2?n.gq(t.u,A)||(i|=256):R.R2(t.q)==3?n.Up(t.u,A)||(i|=4194560):R.R2(t.q)==4?n.Up(t.u,A)||(i|=j0):R.R2(t.q)==5&&(n.Up(t.u,A)||(i|=_0))),R.R2(t.r)!=0&&(D=R.R2(t.r)+2,D>7&&(D-=7),r=n.Yo(t.u,A),(D<=2||D!=r)&&(i|=D<<17)),R.H2(t.e)&&(i|=sn),R.H2(t.f)&&(i|=TA)),n.Ln(t.u,A,WA,false),n.Ln(t.u,A,i,true),n.Sk(t.u,A);},C.OK=function(t,A){t.v=false,R.f3(t.s,new C.RK(t,A));},C.PK=function(t,A,e){this.s=W._1(t,n.Pm(A,A.D[0][e])&&n.Pm(A,A.D[1][e])?"Bond Query Features (Multiple)":"Bond Query Features",this),C.FK(this,A,e);},x(294,1,Tn,C.PK),s.mc=function(t){C.HK(this,t);},s.a=0,s.t=0,s.v=false,o.Aab=N(294),C.RK=function(t,A){this.a=t,this.b=A;},x(402,1,{},C.RK),s.nc=function(){C.JK(this.a,this.b);},o.zab=N(402),C.UK=function(){C.UK=f1,C.TK=j(F(o.Ocb,1),yt,2,6,["None","One electron (duplet)","Two electrons (triplet)","Two electrons (singulet)"]);},C.VK=function(t){var A,e,r,P,i,D,u,Q,l;for(e=j(F(o.Q7,1),H,6,15,[8,-2,8,-2,8]),l=j(F(o.Q7,1),H,6,15,[8,-2,4,-2,12,-2,4,-2,12,-2,4,-2,12,-2,8]),R.e3(t.c,e,l),t.o=R.b3(t.c,1,1),R.A2(t.o,t),R.Y2(t.c,R.a3(t.c,"Atom Label:"),1,1),R.Y2(t.c,t.o,3,1),R.Z2(t.c,R.a3(t.c,"(examples: 'D', 'Li', 'Cys', 'R12', 'R3@C')"),1,3,3,3),t.p=R.b3(t.c,1,1),R.A2(t.p,t),R.Y2(t.c,R.a3(t.c,"Atom Mass:"),1,5),R.Y2(t.c,t.p,3,5),R.Z2(t.c,R.a3(t.c,"(empty for natural abundance)"),1,7,3,7),t.q=R.b3(t.c,1,1),R.A2(t.q,t),R.Y2(t.c,R.a3(t.c,"Abnormal Valence:"),1,9),R.Y2(t.c,t.q,3,9),R.Z2(t.c,R.a3(t.c,"(empty for default valence)"),1,11,3,11),t.a==-1?(r=(n.Rk(),n.Lk)[t.k],R.Z3(t.o,t.n==null?r:t.n+"@"+r),t.g!=0&&R.Z3(t.p,""+t.g),t.j!=-1&&R.Z3(t.q,""+t.j)):(r=n.Il(t.e,t.a),A=n.El(t.e,t.a),R.Z3(t.o,A==null?r:A+"@"+r),n.Ml(t.e,t.a)!=0&&R.Z3(t.p,""+n.Ml(t.e,t.a)),n.zl(t.e,t.a)!=-1&&R.Z3(t.q,""+n.zl(t.e,t.a))),t.b=R._2(t.c),i=C.TK,D=0,u=i.length;D<u;++D)P=i[D],R.Q2(t.b,P);Q=t.a==-1?t.i:n.Pl(t.e,t.a),R.V2(t.b,Q==32?1:Q==48?2:Q==16?3:0),R.Y2(t.c,R.a3(t.c,"Radical State:"),1,13),R.Y2(t.c,t.b,3,13);},C.WK=function(t,A){A.j==1?(t.a!=-1&&(n.Cn(t.e,t.a,t.k),n.tn(t.e,t.a,t.g),n.gn(t.e,t.a,t.j),n.wn(t.e,t.a,t.i),n.mn(t.e,t.a,t.n)),R.c3(t.c)):A.j==0&&(C.YK(t),R.c3(t.c));},C.XK=function(t,A){C.NM(A,t.f);},C.YK=function(t){var A,e,r,P,i,D,u;if(D=R.Y3(t.o),e=null,w.SAb(D).length!=0&&(r=h.hkb(D,X1(64)),r!=-1&&(e=(w.MAb(0,r,w.SAb(D).length),w.SAb(D).substr(0,r)),D=(w.NAb(r+1,w.SAb(D).length+1),w.SAb(D).substr(r+1)))),w.SAb(D).length!=0&&(A=n.po(D,t.d.c),A!=0||h.akb(D,"?"))){if(P=0,w.SAb(R.Y3(t.p)).length!=0)try{if(P=h.Mib(R.Y3(t.p),r1,qt),P<(n.Rk(),n.Pk)[A]-18||P>n.Pk[A]+12){R.g3(t.c,"Your mass is out of range!");return;}}catch(Q){if(Q=C1(Q),Zt(Q,40)){R.g3(t.c,"Your mass is not a number!");return;}else throw st(Q);}if(u=-1,w.SAb(R.Y3(t.q)).length!=0)try{if(u=h.Mib(R.Y3(t.q),r1,qt),u<0||u>15){R.g3(t.c,"Your valence is out of range!");return;}}catch(Q){if(Q=C1(Q),Zt(Q,40)){R.g3(t.c,"Your valence is not a number!");return;}else throw st(Q);}i=R.R2(t.b)==1?32:R.R2(t.b)==2?48:R.R2(t.b)==3?16:0,C.nM(t.d,A,P,u,i,e),t.a!=-1&&(n.fl(t.e,t.a,A,P,u,i),n.mn(t.e,t.a,e)),t.f=true,R.c3(t.c);}},C.ZK=function(t,A){t.f=false,R.f3(t.c,new C.bL(t,A));},C.$K=function(t,A,e,r,P,i,D){C.UK(),this.c=W._1(t,jD,this),this.d=A,this.a=-1,this.k=e,this.g=r,this.j=P,this.i=i,this.n=D,C.VK(this);},C._K=function(t,A,e,r){C.UK(),this.c=W._1(t,jD,this),this.d=A,this.e=e,this.a=r,this.k=n.Tl(this.e,r),this.g=n.Ml(this.e,r),this.j=n.zl(this.e,r),this.i=n.Pl(this.e,r),this.n=n.El(this.e,r),C.VK(this);},x(220,1,Tn,C.$K,C._K),s.mc=function(t){C.WK(this,t);},s.a=0,s.f=false,s.g=0,s.i=0,s.j=0,s.k=0,o.Cab=N(220),C.bL=function(t,A){this.a=t,this.b=A;},x(403,1,{},C.bL),s.nc=function(){C.XK(this.a,this.b);},o.Bab=N(403),U.dL=function(t){this.k=t;},x(324,1,{}),o.Kdb=N(324),ht.eL=function(t,A){U.dL.call(this,A),this.j=t;},x(162,324,{}),s.j=0,o.Uab=N(162),C.fL=function(t,A,e){ht.eL.call(this,A,t),this.a=e;},x(24,162,{24:1},C.fL),s.a=false,o.Dab=N(24),C.hL=function(t,A){U.sj(t.S,A);},C.iL=function(t,A){var e,r,P,i;if(!A||A.q==0)return  false;if(A.r!=0)for(r=n.Vl(t.V,24*t.ab),n.ae(new ht.IN(A),new R.G3(W.o1(t.k)),new ht.pO(0,0,W.m1(t.k),W.l1(t.k)),H1+Mt(r));C.mL(t,A,.2*r);)n._n(A,.5*r,.5*r);if(C.zM(t),n.Ym(A),n._m(A),P=t.V.K,t.V.q==0)n.ol(A,t.V),t.a||n.On(t.V,P),t.eb=d.Math.max(t.eb,4),W.r1(t.k),C.JL(t,new C.fL(t,1,true));else {for(i=t.V.q,n.Yk(t.V,A),t.a||n.On(t.V,P),e=0;e<t.V.q;e++)n.xn(t.V,e,e>=i);t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));}return  true;},C.jL=function(t,A){var e,r,P,i,D,u;for(D=(t.U&4)!=0,u=false,P=false,i=false,e=null,r=0;r<t.V.q;r++)if(n.Pm(t.V,r))if(u){if(i!=At.EH(e,n.Ql(t.V,r),n.Rl(t.V,r))){P=true;break;}}else {if(u=true,!D)break;e=U.xj(t.I,0),i=At.EH(e,n.Ql(t.V,r),n.Rl(t.V,r));}return A?D?P?(C.xL(t,true),true):u?(C.wL(t,true),false):(C.xL(t,false),true):(C.wL(t,u),false):D&&(P||!u);},C.kL=function(t){var A,e;n.bw(t.V,15),A=B(o.Q7,H,6,t.V.q,15,1),e=n.hp(t.V,A,false,true),e=C.WL(t,A,e),C.yM(t,A,e),t.M=A,t.L=n.iw(t.V,A,e);},C.lL=function(t,A,e){var r,P,i,D,u;if(t.V.K){if(n.Jm(t.V,A)||n.Jm(t.V,e)||(r=n.Jl(t.V,A),P=n.Jl(t.V,e),r==null^P==null))return  false;if(r!=null){if(r.length!=P.length)return  false;for(i=0;i<r.length;i++)if(r[i]!=P[i])return  false;}if(D=pt(J(n.Ol(t.V,A),1),0),u=pt(J(n.Ol(t.V,e),1),0),D!=u)return  false;}return n.Tl(t.V,A)==n.Tl(t.V,e);},C.mL=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T;for(r=0,e*=e,u=0;u<A.q;u++){for(l=A.J[u].a,y=A.J[u].b,D=false,Q=0;Q<t.V.q;Q++)if(L=n.Ql(t.V,Q),T=n.Rl(t.V,Q),P=L-l,i=T-y,P*P+i*i<e){D=true;break;}D&&++r;}return r==A.q;},C.nL=function(t){var A,e,r;for(C.kL(t),a.RI(new a.TI(),C.OL(t)),r=B(o.Q7,H,6,t.L.length,15,1),A=0;A<t.V.q;A++)e=t.M[A],n.rn(t.V,A,n.Ll(t.L[e],r[e]),n.Ll(t.V,A)==0||n.Am(t.V,A)),++r[e];},C.oL=function(t,A){switch(A){case 0:C.vL(t);return;case 1:C.zM(t),t.eb=d.Math.max(t.eb,6),W.r1(t.k),C.JL(t,new C.fL(t,1,true));return;case 17:C.kM(t),t.eb=d.Math.max(t.eb,2),W.r1(t.k),C.JL(t,new C.fL(t,1,true));return;}},C.pL=function(t,A){var e,r,P,i;for(r=0,P=0,i=0,e=0;e<t.V.q;e++)(!A||n.Pm(t.V,e))&&(P+=n.Ql(t.V,e),i+=n.Rl(t.V,e),++r);return r>1?new B0.Vgb(P/r,i/r):null;},C.qL=function(t,A,e){var r,P,i,D,u;for(u=B(o.acb,o1,159,e,0,1),D=B(o.Q7,H,6,e,15,1),i=0;i<e;i++)u[i]=new B0.Ugb();for(r=0;r<t.V.q;r++)u[A[r]].a+=n.Ql(t.V,r),u[A[r]].b+=n.Rl(t.V,r),++D[A[r]];for(P=0;P<e;P++)u[P].a/=D[P],u[P].b/=D[P];return u;},C.rL=function(t,A){C.zM(t),n.$k(t.V,t.hb,t.kb,A,0,-1,0,null)&&(t.t!=-1&&(t.t=-1,C.JL(t,new C.fL(t,4,true))),t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));},C.sL=function(t,A){var e,r,P;for(r=0,e=0;e<t.V.q;e++)n.Pm(t.V,e)&&++r;P=r!=0&&r!=t.V.q,(t.U&1)!=0?C.uL(t,A,P):C.tL(t,A,P);},C.tL=function(t,A,e){var r;if(t.eb==6){if(e)for(r=0;r<t.V.q;r++)n.sn(t.V,r,!n.Pm(t.V,r));X.jA(new X.CA(e?4:0),t.V),e&&n.Zm(t.V);}n.Ak(t.G,A,new ht.pO(0,0,W.m1(t.k),W.l1(t.k)),H1|t.T);},C.uL=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;if(e&&t.eb==6)for(k=B(o.Q7,H,6,t.L.length,15,1),i=0;i<t.V.q;i++)T=t.M[i],n.sn(t.L[T],k[T],!n.Pm(t.V,i)),++k[T];for(u=B(o.Zab,R1,35,t.L.length,0,1),c=0;c<t.L.length;c++)t.eb==6&&(X.jA(new X.CA(e?4:0),t.L[c]),n.yq(t.L[c])),Q=new ht.IN(t.L[c]),n.ae(Q,A,null,H1|t.T),u[c]=Q.v;for(S=1.8000000715255737*t.T,D=n.Vl(t.V,24*t.ab),r=(t.U&4)==0?0:t.eb==5?LD*W.m1(t.k):At.DH(U.xj(t.I,0))*t.T/D,v=.5*S,b=0;b<=t.L.length&&((t.U&4)!=0&&b==t.Y&&(At.FH(U.xj(t.I,0),v-S/2,W.l1(t.k)/2,v-S/2+r,W.l1(t.k)/2),v+=r),b!=t.L.length);b++)l=v-u[b].c,L=.5*(W.l1(t.k)-u[b].a)-u[b].d,n._n(t.L[b],l,L),v+=S+u[b].b;for(n.Ak(t.G,A,new ht.pO(0,0,W.m1(t.k),W.l1(t.k)),H1|t.T),g=B(o.Q7,H,6,t.L.length,15,1),P=0;P<t.V.q;P++)y=t.M[P],n.zn(t.V,P,n.Ql(t.L[y],g[y])),n.An(t.V,P,n.Rl(t.L[y],g[y])),++g[y];n.yq(t.V);},C.vL=function(t){var A;t.I&&(w.kAb(t.I.a,0),t.eb=d.Math.max(t.eb,1),W.r1(t.k)),C.zM(t),A=t.V.K,n._v(t.V),n.On(t.V,A),t.cb.q!=0?(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true))):(t.eb=d.Math.max(t.eb,1),W.r1(t.k));},C.wL=function(t,A){return t.V.q!=0&&t.r?R.K2(t.r,A?C.PL(t.V):n.fw(t.V)):false;},C.xL=function(t,A){var e;return e=A?C.QL(t):C.OL(t),e&&t.r?R.L2():false;},C.yL=function(t,A,e){var r;return n.rl(t.V,A,e)?(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)),true):(r=C.HL(t,A,e),r?(U.Fj(t.I,r),t.v=null,t.eb=d.Math.max(t.eb,1),W.r1(t.k),true):false);},C.zL=function(t){return t.t!=-1?(n.ql(t.V,t.t),t.t=-1,t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)),true):t.u!=-1?(n.ul(t.V,t.u),t.u=-1,t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)),true):t.v&&!t.v.j?(U.Fj(t.I,t.v),t.v=null,t.eb=d.Math.max(t.eb,1),W.r1(t.k),true):false;},C.AL=function(t,A){var e,r,P;if(t.t!=-1&&(R.F3(A,(P=R.B3(A),P!=0?P:kr)),C.TL(t,A,t.t),t.w==19&&(r=n.Ll(t.V,t.t),r!=0)))for(e=0;e<t.V.f;e++)e!=t.t&&n.Ll(t.V,e)==r&&C.TL(t,A,e);t.u!=-1&&(R.F3(A,(P=R.B3(A),P!=0?P:kr)),C.UL(t,A,t.u)),t.v&&t.v.Mb(A);},C.BL=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b;for(i=0,e=0;e<t.V.q;e++)n.Pm(t.V,e)&&++i;for(c=t.V.q,b=t.V.r,t.gb=U.yrb(t.gb,t.gb.length+i),t.jb=U.yrb(t.jb,t.jb.length+i),D=B(o.Q7,H,6,t.V.q,15,1),Q=n.cn(t.V,1),l=n.cn(t.V,2),r=0;r<c;r++)n.Pm(t.V,r)&&(y=t.V.q,t.gb[y]=t.gb[r],t.jb[y]=t.jb[r],D[r]=y,n.ll(t.V,t.V,r,Q,l));for(u=0;u<b;u++)n.Qm(t.V,u)&&n.nl(t.V,t.V,u,Q,l,D,false);for(P=0;P<c;P++)n.xn(t.V,P,false);for(A=c;A<t.V.q;A++)n.rn(t.V,A,0,false);if(t.I)for(L=t.I.a.length-1;L>=0;L--)T=U.xj(t.I,L),T.g&&!Zt(T,61)&&U.sj(t.I,T.Gb());},C.CL=function(t,A,e){C.wN(new C.xN(t._,A),new C.UM(t,A,e));},C.DL=function(t){var A,e,r;if(null.zd()||null.zd())C.jL(t,true);else if(null.zd())t.r&&R.N2();else if(null.zd())C.GM(t,null.zd());else if(null.zd())C.hM(t);else if(!null.zd())if(null.zd())C.nL(t),C.EM(t,d.Math.max(t.eb,1));else if(null.zd())C.jM(t,true,true);else if(null.zd())C.jM(t,false,true);else if(null.zd())C.jM(t,true,false);else if(null.zd())C.cM(t);else if(null.zd())C.gL=!C.gL,C.gL&&C.nL(t);else if(null.zd())C.KL(t,true);else if(null.zd())C.KL(t,false);else if(null.zd())C.lM(t);else if(null.zd())W.j2(t._,pD,MD);else if(null.zd()){if(null.zd(),A=h.Mib(null.zd(),r1,qt),e=h.Mib(null.zd(),r1,qt),n.Pm(t.V,A))for(r=0;r<t.V.f;r++)n.Pm(t.V,r)&&n.kn(t.V,r,e);else n.kn(t.V,A,e);}else null.zd()?(t.H=6,t.eb=d.Math.max(t.eb,1),W.r1(t.k)):null.zd()?(t.H=256,t.eb=d.Math.max(t.eb,1),W.r1(t.k)):null.zd()&&C.oM(t,t.w==19?16:0);},C.EL=function(t,A){var e,r,P,i,D,u,Q,l,L,y;if(!t.W){if(l=A.b,A.j==1)if(l==-3)t.Z=true,C.FM(t);else if(l==-2)t.d=true,C.FM(t);else if(l==-1)C.FM(t);else if(A.a)l==122?(C.kM(t),t.eb=d.Math.max(t.eb,2),W.r1(t.k),C.JL(t,new C.fL(t,1,true))):l==99?C.jL(t,true):l==118&&((t.U&4)!=0&&t.r&&R.N2(),C.hM(t));else if(l==-4)t.w!=19&&(C.zM(t),C.zL(t)||n.wl(t.V)&&(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true))));else if(l==-6||t.t==-1&&l==63){W.j2(t._,pD,MD);return;}else if(l==-8)w.SAb(t.i.a).length!=0&&(C.GL(t,t.i.a),h.whb(t.i,0));else if(t.u!=-1)l==113&&t.V.K?C.wM(t,t.u):l==118?n.cl(t.V,t.u,3,false,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true))):l>=52&&l<=55?n.cl(t.V,t.u,l-48,false,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true))):l==97||l==98?n.cl(t.V,t.u,6,true,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true))):(P=l==48?(C.zM(t),n.il(t.V,t.u,16)):l==49?(C.zM(t),n.il(t.V,t.u,1)):l==50?(C.zM(t),n.il(t.V,t.u,2)):l==51?(C.zM(t),n.il(t.V,t.u,4)):l==117?(C.zM(t),n.il(t.V,t.u,257)):l==100?(C.zM(t),n.il(t.V,t.u,129)):l==99?(C.zM(t),n.il(t.V,t.u,386)):l==109&&(C.zM(t),n.il(t.V,t.u,16)),P&&(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true))));else if(t.t!=-1){if(Q=w.SAb(t.i.a).length==0,Q?t.K=l:(t.K==108&&(h.whb(t.i,0),h.Ekb(t.i,76)),t.K=-1),Q&&l==108)h.Ikb(t.i,"Cl"),t.eb=d.Math.max(t.eb,1),W.r1(t.k);else if(Q&&(l==43||l==45))C.zM(t),n.hl(t.V,t.t,l==43)&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));else if(Q&&l==46)C.zM(t),y=n.Pl(t.V,t.t)==32?0:32,n.wn(t.V,t.t,y),t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));else if(Q&&l==58)C.zM(t),y=n.Pl(t.V,t.t)==48?16:n.Pl(t.V,t.t)==16?0:48,n.wn(t.V,t.t,y),t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));else if(Q&&l==113&&t.V.K)C.vM(t,t.t);else if(Q&&t.V.K&&(l==120||l==88))L=j(F(o.Q7,1),H,6,15,[9,17,35,53]),n.pn(t.V,t.t,L),t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));else if(Q&&l==63)C.zM(t),n.fl(t.V,t.t,0,0,-1,0)&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));else if(Q&&l>48&&l<=57){if(n.jp(t.V,t.t)>0){for(C.zM(t),i=l-47,e=t.t,D=t.V.q-t.V.f,u=1;u<i&&(C.BM(t,e),r=n.Tk(t.V,t.ib,t.lb,0),r!=-1);u++)n.Vk(t.V,e,r),e=r-D,n.bw(t.V,1);t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));}}else !Q&&l==-7?(h.whb(t.i,0),t.eb=d.Math.max(t.eb,1),W.r1(t.k)):!Q&&l==-5?(h.whb(t.i,w.SAb(t.i.a).length-1),t.eb=d.Math.max(t.eb,1),W.r1(t.k)):l>=65&&l<=90||l>=97&&l<=122||l>=48&&l<=57||l==45?(h.Ekb(t.i,l&e1),t.eb=d.Math.max(t.eb,1),W.r1(t.k)):(l==10||l==13)&&C.GL(t,t.i.a);}else l==104||l==118?(t.U&7)==0&&C.KL(t,l==104):t.db&&C.cN(t.db,A);A.j==2&&(l==-3?(t.Z=false,C.FM(t)):l==-2?(t.d=false,C.FM(t)):l==-1?C.FM(t):t.db&&C.cN(t.db,A));}},C.FL=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S;if(A.j==1){if(t.t!=-1&&w.SAb(t.i.a).length!=0&&C.GL(t,t.i.a),h.whb(t.i,0),A.e){C.SL(t);return;}if(A.b==1){if(A.c==2)return;t.W=true,C.FM(t),C.eM(t,A);}}if(A.j==2){if(A.e){C.SL(t);return;}A.b==1&&(t.W=false,C.FM(t),C.fM(t));}if(A.j==3&&A.b==1&&A.c==2){C.RL(t,A.g,A.i);return;}if(A.j==4&&(W.f2(t._),C.FM(t)),A.j==6&&(t.W=false,v=A.g,S=A.i,C.DM(t,v,S,false)&&W.r1(t.k),C.FM(t)),A.j==7){switch(t.W=true,t.ib=A.g,t.lb=A.i,T=C.DM(t,t.ib,t.lb,true),t.X){case 2:C.AM(t)&&(T=true);break;case 1:(t.ib-t.hb)*(t.ib-t.hb)+(t.lb-t.kb)*(t.lb-t.kb)<100&&C.BM(t,t.e),T=true;break;case 3:n.zn(t.V,t.e,t.gb[t.e]+t.ib-t.hb),n.An(t.V,t.e,t.jb[t.e]+t.lb-t.kb),t.f!=-1&&(n.zn(t.V,t.f,t.gb[t.f]+t.ib-t.hb),n.An(t.V,t.f,t.jb[t.f]+t.lb-t.kb)),t.eb=d.Math.max(t.eb,2),W.r1(t.k);break;case 9:C.BL(t),t.X=4;case 4:if(t.I)for(i=new U.rrb(t.I);i.a<i.c.a.length;)P=U.qrb(i),P.g&&P.Ob(t.ib,t.lb);for(u=0;u<t.V.q;u++)n.Pm(t.V,u)&&(n.zn(t.V,u,t.gb[u]+t.ib-t.hb),n.An(t.V,u,t.jb[u]+t.lb-t.kb));t.eb=d.Math.max(t.eb,2),W.r1(t.k);break;case 11:U.sj(t.I,t.v.Gb()),t.X=10;case 10:t.v.Ob(t.ib,t.lb),t.eb=d.Math.max(t.eb,2),W.r1(t.k);break;case 7:for(c=false,r=0;r<t.V.q&&!c;r++)c=n.Pm(t.V,r);if(b=false,t.I)for(Q=0;Q<t.I.a.length&&!b;Q++)b=U.xj(t.I,Q).g;if(L=d.Math.abs(t.lb-t.kb)<20?1:d.Math.exp((t.lb-t.kb)/100),e=d.Math.abs(t.ib-t.hb)<20?0:(t.ib-t.hb)/50,g=c||b,t.I&&(!g||b)){for(Q=0;Q<t.I.a.length;Q++)(!g||U.xj(t.I,Q).g)&&U.xj(t.I,Q).Pb(L,e);t.eb=d.Math.max(t.eb,2),W.r1(t.k);}(!g||c)&&(n.bo(t.V,L,e,g),t.eb=d.Math.max(t.eb,2),W.r1(t.k));break;case 5:case 6:if(k=null,t.X==5){if(d.Math.abs(t.ib-ht.eO(t.R,t.R.a-1))<3&&d.Math.abs(t.lb-ht.fO(t.R,t.R.a-1))<3)break;ht.gO(t.R),ht.dO(t.R,t.ib,t.lb),ht.dO(t.R,t.hb,t.kb),k=t.R;}else k=new ht.pO(d.Math.min(t.hb,t.ib),d.Math.min(t.kb,t.lb),d.Math.abs(t.ib-t.hb),d.Math.abs(t.lb-t.kb));if(t.I)for(Q=0;Q<t.I.a.length;Q++)y=U.xj(t.I,Q),l=n.ke(y,k),(!t.O||!t.Q[Q])&&l!=y.g&&(y.g=l,t.eb=d.Math.max(t.eb,1));for(D=0;D<t.V.q;D++)l=k.Hb(Mt(n.Ql(t.V,D)),Mt(n.Rl(t.V,D))),(!t.O||!t.P[D])&&l!=n.Pm(t.V,D)&&(n.xn(t.V,D,l),t.eb=d.Math.max(t.eb,1));T=true;break;case 8:T=true;}T&&W.r1(t.k);}},C.GL=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b;if(h.whb(t.i,0),r=n.po(A,t.c),r!=0){if(C.zM(t),y=r==1^n.Tl(t.V,t.t)==1,n.fl(t.V,t.t,r,0,-1,0)){y&&(t.t=-1,C.JL(t,new C.fL(t,4,true))),t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));return;}t.eb=d.Math.max(t.eb,0),W.r1(t.k);}if(t.j!=null&&w.SAb(t.j).length!=0&&(A=t.j),c=(n.Rs(),L=(!n.Qs&&n.Ss(),U.eqb(n.Qs,h.mkb(w.SAb(A).toLowerCase(),"-"))),L==null?null:n.Pq(new n._q(false),L)),c){for(C.zM(t),Q=new n.tw(),n.xo(Q,t.V,t.t,null),T=n.Vl(Q,t.ab*24),D=Q.q,e=0;e<Q.q;e++)Q.u[e]|=r0;for(n.dl(Q,c,0),X.jA(new X.CA(4),Q),P=n.Ql(t.V,t.t)-T*Q.J[0].a,i=n.Rl(t.V,t.t)-T*Q.J[0].b,u=t.V.q,n.dl(t.V,c,t.t),b=t.V.q-u,l=0;l<b;l++)n.zn(t.V,u+l,T*Q.J[D+l].a+P),n.An(t.V,u+l,T*Q.J[D+l].b+i);n.yq(t.V),t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));}},C.HL=function(t,A,e){var r,P;if(t.I){for(P=new U.rrb(t.I);P.a<P.c.a.length;)if(r=U.qrb(P),r.Hb(A,e))return r;}return null;},C.IL=function(t){var A,e,r,P,i,D;for(i=-1,D=s0,A=0;A<t.V.q;A++)r=t.hb-n.Ql(t.V,A),P=t.kb-n.Rl(t.V,A),e=d.Math.sqrt(r*r+P*P),e<24&&D>e&&(D=e,i=t.M[A]);return i;},C.JL=function(t,A){var e;e=A.j,(e&t.J)==0&&(W.h2(t._,new C.VM(t,e,A)),t.J|=e);},C.KL=function(t,A){var e,r,P,i,D;for(D=false,r=0;r<t.V.q;r++)if(n.Pm(t.V,r)){D=true;break;}if(i=C.pL(t,D),i){for(C.zM(t),e=0;e<t.V.q;e++)(!D||n.Pm(t.V,e))&&(A?n.zn(t.V,e,2*i.a-n.Ql(t.V,e)):n.An(t.V,e,2*i.b-n.Rl(t.V,e)));for(P=0;P<t.V.r;P++)(!D||n.Qm(t.V,P))&&(n.im(t.V,P)==257?n.Mn(t.V,P,129):n.im(t.V,P)==129&&n.Mn(t.V,P,257));t.eb=d.Math.max(t.eb,1),W.r1(t.k);}},C.LL=function(t,A){return n.po(A,t.c)!=0?1:(t.j=n.Ts(A),t.j==null?C.VL(A)?3:4:w.SAb(t.j).length==0?3:2);},C.ML=function(t,A){var e,r,P;if(e=n.Zl(t.V,0,A),n.Nl(t.V,e)!=0)return n.ym(t.V,e)||n.Nl(t.V,e)!=1&&n.Nl(t.V,e)!=2?-1:e;if(n.Qo(t.V,e)==1){for(P=0;P<n.$o(t.V,e);P++)if(n.ap(t.V,e,P)==2&&(r=n.Zo(t.V,e,P),n.Qo(t.V,r)==2&&(n.Nl(t.V,r)==1||n.Nl(t.V,r)==2)))return r;}return  -1;},C.NL=function(t,A){var e;return e=n.Ho(t.V,n.Zl(t.V,0,A)),e!=-1&&n.gm(t.V,e)!=1&&n.gm(t.V,e)!=2&&(e=-1),e;},C.OL=function(t){var A,e;if((t.U&4)==0||t.L==null)return null;for(e=new At.zH(),A=0;A<t.L.length;A++)A<t.Y?At.dH(e,t.L[A]):At.bH(e,t.L[A]);return e;},C.PL=function(t){var A,e,r,P,i,D,u;for(r=0,e=0;e<t.q;e++)(t.u[e]&512)!=0&&++r;if(r==0)return null;for(i=0,P=0;P<t.r;P++)(t.u[t.D[0][P]]&t.u[t.D[1][P]]&512)!=0&&++i;for(u=B(o.vfb,Tt,6,t.q,16,1),A=0;A<t.q;A++)u[A]=(t.u[A]&512)!=0;return D=new n.uw(r,i),n.Co(t,D,u,false,null),D;},C.QL=function(t){var A,e,r;for(e=new At.zH(),A=0;A<t.L.length;A++)r=C.PL(t.L[A]),r&&(A<t.Y?(U.sj(e.g,r),e.d=-1):(U.sj(e.f,r),e.d=-1));return e;},C.RL=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v;if(i=n.xl(t.V,A,e),D=n.yl(t.V,A,e),t.w==2){if(t.V.K){if(i!=-1)C.vM(t,i);else if(D!=-1)C.wM(t,D);else if(t.v){if(!t.Z){for(T=0;T<t.V.q;T++)n.xn(t.V,T,false);for(P=new U.rrb(t.I);P.a<P.c.a.length;)r=U.qrb(P),r.g=false;}n.ne(t.v,true),t.eb=d.Math.max(t.eb,1),W.r1(t.k);}}else if(v=-1,i!=-1?v=i:D!=-1&&(v=n.Zl(t.V,0,D)),v!=-1||t.v){if(!t.Z){for(T=0;T<t.V.q;T++)n.xn(t.V,T,false);if(t.I)for(l=new U.rrb(t.I);l.a<l.c.a.length;)Q=U.qrb(l),Q.g=false;}if(v!=-1){if((t.U&1)!=0)for(L=t.M[v],T=0;T<t.V.q;T++)t.M[T]==L&&n.xn(t.V,T,true);else for(y=n.fp(t.V,v,false),T=0;T<y.length;T++)n.xn(t.V,y[T],true);}else n.ne(t.v,true);t.eb=d.Math.max(t.eb,1),W.r1(t.k);}}else if(t.w==18&&(L=-2,(t.U&1)!=0&&(L=C.IL(t)),L!=-1)){for(k=qt,g=r1,c=0;c<t.V.q;c++)(L==-2||t.M[c]==L)&&(k>n.Ql(t.V,c)&&(k=n.Ql(t.V,c)),g<n.Ql(t.V,c)&&(g=n.Ql(t.V,c)));if(g>k){for(u=(g+k)/2,b=0;b<t.V.q;b++)(L==-2||t.M[b]==L)&&n.zn(t.V,b,2*u-n.Ql(t.V,b));for(T=0;T<t.V.r;T++)if(L==-2||t.M[n.Zl(t.V,0,T)]==L)switch(n.im(t.V,T)){case 257:n.Mn(t.V,T,129);break;case 129:n.Mn(t.V,T,257);}}t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true));}},C.SL=function(t){t.r&&(C.jL(t,false),null.zd(),null.zd(),(t.U&4)!=0&&(null.zd(),null.zd()),(t.U&4)!=0&&null.zd()),(t.U&4)!=0&&t.w==19&&(null.zd(),null.zd(),null.zd(),null.zd(),null.zd(),null.zd()),t.w==18&&(null.zd(),null.zd(),null.zd()),t.g&&t.t!=-1&&(n.Cl(t.V,t.t),null.zd(),null.zd(),null.zd(),null.zd(),null.zd(),null.zd(),null.zd(),null.zd(),null.zd(),null.zd()),null.zd();},C.TL=function(t,A,e){var r,P,i;r=Mt(MP*n.Ul(t.V)),P=Mt(n.Ql(t.V,e)),i=Mt(n.Rl(t.V,e)),R.t3(A,P-r,i-r,2*r);},C.UL=function(t,A,e){var r,P,i,D,u,Q;P=Mt(MP*n.Ul(t.V)),i=Mt(n.Ql(t.V,n.Zl(t.V,0,e))),u=Mt(n.Rl(t.V,n.Zl(t.V,0,e))),D=Mt(n.Ql(t.V,n.Zl(t.V,1,e))),Q=Mt(n.Rl(t.V,n.Zl(t.V,1,e))),r=R.A3(A),R.E3(A,P),R.p3(A,i,u,D,Q),R.E3(A,r);},C.VL=function(t){var A;if(w.SAb(t).length<3){for(A=1;A<(n.Rk(),n.Lk).length;A++)if(h.pkb(n.Lk[A],t))return  true;}return  false;},C.WL=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G;if(e<2)return e;for(q=B(o.vfb,o1,12,e,0,2),b=1;b<e;b++)q[b]=B(o.vfb,Tt,6,b,16,1);for(D=n.Vl(t.V,24*t.ab),P=1;P<t.V.q;P++)for(i=0;i<P;i++)Q=n.Ql(t.V,i)-n.Ql(t.V,P),l=n.Rl(t.V,i)-n.Rl(t.V,P),u=d.Math.sqrt(Q*Q+l*l),u<ce*D&&(y=A[P],T=A[i],y!=T&&(y>T?q[y][T]=true:q[T][y]=true));for(G=B(o.Q7,H,6,e,15,1),L=0;L<e;L++)G[L]=L;for(E=0,c=1;c<e;c++)for(v=0;v<c;v++)if(q[c][v]&&(g=G[c],k=G[v],g!=k))for(++E,V=d.Math.min(g,k),p=d.Math.max(g,k),S=0;S<e;S++)G[S]==p?G[S]=V:G[S]>p&&--G[S];for(r=0;r<t.V.q;r++)A[r]=G[A[r]];return e-E;},C.XL=function(t,A){A&&(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));},C.YL=function(t,A){A&&(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));},C.ZL=function(t,A,e,r){r&&A!=-1&&(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true))),e.nc();},C.$L=function(t){C.zM(t),t.eb=d.Math.max(t.eb,3),W.r1(t.k);},C._L=function(t,A,e){var r,P;for(P=false,r=0;r<w.SAb(A.e).length;r++)if(!n2(h.Ujb(A.e,r))){P=true;break;}P||U.Fj(t.I,A),W.r1(t.k),C.$L(e.a);},C.aM=function(t,A,e){var r,P;for(t.J&=~A,P=new U.rrb(t.S);P.a<P.c.a.length;)r=U.qrb(P),r.mc(e);},C.bM=function(t,A,e,r){return (t.U&6)!=0&&e[1]!=r[1]?e[1]==0?-1:1:JA(A[e[0]].a+A[e[0]].b,A[r[0]].a+A[r[0]].b);},C.cM=function(t){var A,e;for(e=false,A=0;A<t.V.f;A++)n.Ll(t.V,A)!=0&&n.Am(t.V,A)&&(e||(C.zM(t),e=true),n.rn(t.V,A,n.Ll(t.V,A),false));e&&C.EM(t,d.Math.max(t.eb,1));},C.dM=function(t){t.eb=d.Math.max(t.eb,4),W.r1(t.k),C.JL(t,new C.fL(t,1,false));},C.eM=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b;switch(t.hb=A.g,t.kb=A.i,t.w){case 18:if(c=t.hb,b=t.kb,t.e=n.xl(t.V,t.hb,t.kb),t.e!=-1&&(t.hb=n.Ql(t.V,t.e),t.kb=n.Rl(t.V,t.e)),n.co(t.V,c,b),t.I)for(u=new U.rrb(t.I);u.a<u.c.a.length;)D=U.qrb(u),D.Qb(c,b);C.zM(t),t.X=7;break;case 2:if(t.X=0,t.e=n.xl(t.V,t.hb,t.kb),t.e!=-1&&(t.f=-1,t.hb=n.Ql(t.V,t.e),t.kb=n.Rl(t.V,t.e),n.Pm(t.V,t.e)?t.X=t.Z?9:4:t.X=3),t.X==0&&(P=n.yl(t.V,t.hb,t.kb),P!=-1&&(t.e=n.Zl(t.V,0,P),t.f=n.Zl(t.V,1,P),n.Qm(t.V,P)?t.X=t.Z?9:4:t.X=3)),t.X==0&&t.v&&(t.v.g?t.X=t.Z?9:4:t.X=t.Z&&!Zt(t.v,61)?11:10),t.X!=0){for(t.gb=B(o.O7,Ft,6,t.V.q,15,1),t.jb=B(o.O7,Ft,6,t.V.q,15,1),Q=0;Q<t.V.q;Q++)t.gb[Q]=n.Ql(t.V,Q),t.jb[Q]=n.Rl(t.V,Q);if(t.I)for(u=new U.rrb(t.I);u.a<u.c.a.length;)D=U.qrb(u),n.pe(D,t.hb,t.kb);C.zM(t);break;}if(A.a?t.X=6:(t.R=new ht.hO(),ht.dO(t.R,t.hb,t.kb),ht.dO(t.R,t.hb,t.kb),t.X=5),t.P=B(o.vfb,Tt,6,t.V.q,16,1),t.I&&(t.Q=B(o.vfb,Tt,6,t.I.a.length,16,1)),t.O=A.f,t.O){for(l=0;l<t.V.q;l++)t.P[l]=n.Pm(t.V,l);if(t.I)for(Q=0;Q<t.I.a.length;Q++)t.Q[Q]=U.xj(t.I,Q).g;}else {for(y=false,l=0;l<t.V.q;l++)n.Pm(t.V,l)&&(n.xn(t.V,l,false),y=true);if(t.I)for(Q=0;Q<t.I.a.length;Q++)U.xj(t.I,Q).g&&(n.ne(U.xj(t.I,Q),false),y=true);y&&(t.eb=1,W.r1(t.k));}break;case 4:C.zM(t),C.yL(t,t.hb,t.kb);break;case 3:T=n.xl(t.V,t.hb,t.kb),T!=-1&&(C.zM(t),n.ln(t.V,T,!n.wm(t.V,T)),t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 101:case 103:case 102:t.u!=-1&&C.iM(t,t.u)&&(C.zM(t),C.pM(t,t.u,t.w==101?0:t.w==103?1:2),t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 5:case 6:case 23:if(t.e=n.xl(t.V,t.hb,t.kb),t.e==-1){if(r=n.yl(t.V,t.hb,t.kb),r!=-1){C.zM(t),i=511,t.w==6?i=257:t.w==23&&(i=129),n.il(t.V,r,i)&&(t.eb=d.Math.max(t.eb,1),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;}}else {if(n.Mo(t.V,t.e)==8)return;t.hb=n.Ql(t.V,t.e),t.kb=n.Rl(t.V,t.e);}t.X=1,C.BM(t,t.e),W.r1(t.k);break;case 22:if(t.e=n.xl(t.V,t.hb,t.kb),t.e!=-1){if(n.Mo(t.V,t.e)==8)return;t.hb=n.Ql(t.V,t.e),t.kb=n.Rl(t.V,t.e);}t.X=2,t.q=0;break;case 7:C.zM(t),n.al(t.V,t.hb,t.kb,3,false,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 24:C.zM(t),n.al(t.V,t.hb,t.kb,4,false,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 8:C.zM(t),n.al(t.V,t.hb,t.kb,5,false,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 25:C.zM(t),n.al(t.V,t.hb,t.kb,6,false,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 9:C.zM(t),n.al(t.V,t.hb,t.kb,7,false,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 26:C.zM(t),n.al(t.V,t.hb,t.kb,6,true,n.Vl(t.V,24*t.ab))&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 10:C.zM(t),n.gl(t.V,t.hb,t.kb,true)&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 27:C.zM(t),n.gl(t.V,t.hb,t.kb,false)&&(t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true)));break;case 16:C.rL(t,1);break;case 11:C.rL(t,6);break;case 12:C.rL(t,7);break;case 13:C.rL(t,8);break;case 28:C.rL(t,14);break;case 29:C.rL(t,15);break;case 30:C.rL(t,16);break;case 14:C.rL(t,9);break;case 31:C.rL(t,17);break;case 15:C.rL(t,35);break;case 32:C.rL(t,53);break;case 33:A.d||A.a?(e=n.xl(t.V,t.hb,t.kb),e!=-1&&C.xM(t,e,new C.PM())):(C.zM(t),n.$k(t.V,t.hb,t.kb,t.F,t.B,t.D,t.C,t.A)&&(t.t=-1,t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true))));break;case 19:t.e=n.xl(t.V,t.hb,t.kb),t.e!=-1&&t.e<t.V.f&&(!t.V.K||!n.Jm(t.V,t.e))&&(t.hb=n.Ql(t.V,t.e),t.kb=n.Rl(t.V,t.e),t.X=8);break;case 21:L=null,t.v?Zt(t.v,94)&&(L=t.v):(L=new n.ax(),n.Ww(L,t.hb,t.kb),U.sj(t.I,L)),C.CL(t,L,new C.SM(t));}},C.fM=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g;switch(T=t.X,t.X=0,T){case 1:if(b=n.xl(t.V,t.ib,t.lb),b!=-1&&n.Mo(t.V,b)==8)return;C.zM(t),t.e==-1&&(t.e=n.Tk(t.V,t.hb,t.kb,0)),b==-1&&(b=n.Tk(t.V,t.ib,t.lb,0)),t.e!=-1&&b!=-1&&(P=n.Lm(t.V,t.e)||n.Lm(t.V,b)?16:1,t.w==6&&(P=257),t.w==23&&(P=129),n._k(t.V,t.e,b,P)),t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));break;case 2:if(C.zM(t),t.q>0&&(t.e==-1&&(t.e=n.Tk(t.V,t.hb,t.kb,0)),t.n[0]==-1&&(t.n[0]=n.Tk(t.V,t.o[0],t.p[0],0)),t.n[0]!=-1&&n.Vk(t.V,t.e,t.n[0])),t.q>1)for(u=1;u<t.q;u++)t.n[u]==-1&&(t.n[u]=n.Tk(t.V,t.o[u],t.p[u],0)),t.n[u]!=-1&&n.Vk(t.V,t.n[u-1],t.n[u]);t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));break;case 3:case 4:case 7:t.eb=d.Math.max(t.eb,3),W.r1(t.k),C.JL(t,new C.fL(t,1,true));break;case 10:t.eb=d.Math.max(t.eb,3),W.r1(t.k);break;case 5:case 6:for(c=false,D=0;D<t.V.q;D++)if(t.P[D]!=n.Pm(t.V,D)){c=true;break;}W.r1(t.k),c&&C.JL(t,new C.fL(t,2,true));break;case 8:if(y=false,r=t.t,r!=-1&&t.V.K&&n.Jm(t.V,r)&&(r=-1),l=n.Ll(t.V,t.e),r==-1){if(C.zM(t),l!=0){for(y=true,g=false,A=0;A<t.V.f;A++)n.Ll(t.V,A)==l&&(g=g|n.Am(t.V,A),n.rn(t.V,A,0,false));C.gL&&!g&&C.nL(t);}}else {if(C.zM(t),y=true,g=false,t.e==r)for(Q=n.Ll(t.V,t.e),A=0;A<t.V.f;A++)n.Ll(t.V,A)==Q&&(g=g|n.Am(t.V,A),n.rn(t.V,A,0,false));else {for(L=n.Ll(t.V,r),e=0;e<t.V.f;e++)(n.Ll(t.V,e)==l||n.Ll(t.V,e)==L)&&n.rn(t.V,e,0,false);for(i=1,A=0;A<t.V.f;A++)n.Ll(t.V,A)==i&&(++i,A=-1);n.rn(t.V,t.e,i,false),n.rn(t.V,r,i,false);}C.gL&&!g&&C.nL(t);}y&&C.EM(t,d.Math.max(t.eb,1)),W.r1(t.k);}},C.gM=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E,q,V,G,K,O,tt,Qt,z,it,lt,ut,Dt,ot,bt,Ut,wt;if((t.fb!=W.m1(t.k)||t.N!=W.l1(t.k))&&(t.fb=W.m1(t.k),t.N=W.l1(t.k),t.eb<3&&(t.eb=3)),P=R.w3(A),u=R.z3(A),R.F3(A,P),R.v3(A,0,0,t.fb,t.N),(t.U&4)!=0&&t.I.a.length==0&&(c=.5*t.fb,b=.5*t.N,i=.03999999910593033*t.fb,e=new At.GH(),At.FH(e,c-i,b,c+i,b),e.j=true,U.sj(t.I,e)),L=false,t.eb!=0){switch((t.U&1)!=0&&t.eb!=5&&C.kL(t),t.G=(t.U&4)!=0?new n.Dk(new At.BH(t.L,t.Y),t.I):(t.U&2)!=0?new n.Ek(t.L,t.Y,t.I):(t.U&1)!=0?new n.Fk(t.L,t.I):new n.Ck(t.V,t.I),n.yk(t.G,u,P),n.vk(t.G,t.T),n.zk(t.G,(t.U&1)==0?0:(y=dr,h.akb(w.SAb(y).substr(0,18),On)||h.akb(w.SAb(y).substr(0,17),xn)||h.akb(w.SAb(y).substr(0,14),Nn)?Lt.u4(P,he):(Lt.t4(),P&G0|Mt(d.Math.round(he*((P&W0)>>16)))<<16|Mt(d.Math.round(he*((P&Y0)>>8)))<<8|Mt(d.Math.round(he*(P&255)))))),n.xk(t.G,t.$),n.wk(t.G,t.H|8|(t.w==19?80:0)),t.eb){case 6:case 4:case 5:C.sL(t,A);break;case 3:q=n.Ak(t.G,A,new ht.pO(0,0,t.fb,t.N),0),q&&(t.U&1)!=0&&n.vi(q,t.V);break;case 2:V=n.Bk(t.G,A,new ht.pO(0,0,t.fb,t.N),0),L=!!V&&!(V.c==1&&V.a==0&&V.b==0);}t.eb=0;}switch(t.G&&n.tk(t.G,A),L||C.AL(t,A),t.G&&(n.uk(t.G,A),n.sk(t.G,A)),t.t!=-1&&w.SAb(t.i.a).length!=0&&(O=Mt(n.Ql(t.V,t.t)),ut=Mt(n.Rl(t.V,t.t)),k=t.i.a,G=C.LL(t,k),R.F3(A,G==1?u:G==2?-16776961:Ie),G==1?k=(n.Rk(),n.Lk)[n.po(k,t.c)]:G==2&&(k=h.rkb(t.j,0,w.SAb(k).length)),D=3*R.y3(A)/2|0,R.D3(A,D,false,false),R.s3(A,O,ut,k),G==4&&(R.F3(A,fA),R.m3(A,O+R.x3(A,k).b/2,ut+D,"<unknown>")),G==2&&(R.F3(A,Ie),R.s3(A,O+R.x3(A,k).b,ut,h.qkb(t.j,w.SAb(k).length)))),R.F3(A,u),t.X){case 1:switch(tt=Mt(t.hb),Dt=Mt(t.kb),t.t==-1||t.t==t.e?(Qt=Mt(t.ib),ot=Mt(t.lb)):(Qt=Mt(n.Ql(t.V,t.t)),ot=Mt(n.Rl(t.V,t.t))),t.w){case 5:R.p3(A,tt,Dt,Qt,ot);break;case 6:z=(Dt-ot)/9|0,bt=(Qt-tt)/9|0,g=new ht.iO(3),ht.dO(g,tt,Dt),ht.dO(g,Qt+z,ot+bt),ht.dO(g,Qt-z,ot-bt),R.u3(A,new R.U3(g));break;case 23:for(z=Qt-tt,bt=ot-Dt,l=2;l<17;l+=2)it=tt+(l*z/17|0)-(l*bt/128|0),Ut=Dt+(l*bt/17|0)+(l*z/128|0),lt=tt+(l*z/17|0)+(l*bt/128|0),wt=Dt+(l*bt/17|0)-(l*z/128|0),R.p3(A,it,Ut,lt,wt);}break;case 2:if(t.q>0&&R.p3(A,Mt(t.hb),Mt(t.kb),Mt(t.o[0]),Mt(t.p[0])),t.q>1)for(l=1;l<t.q;l++)R.p3(A,Mt(t.o[l-1]),Mt(t.p[l-1]),Mt(t.o[l]),Mt(t.p[l]));break;case 5:R.F3(A,(S=(p=R.B3(A),p!=0?p:kr),Lt.w4(S,(T=dr,h.akb(w.SAb(T).substr(0,w.SAb(On).length),On)||h.akb(w.SAb(T).substr(0,w.SAb(xn).length),xn)||h.akb(w.SAb(T).substr(0,w.SAb(Nn).length),Nn)?RD:CD)))),R.q3(A,new R.U3(t.R)),R.F3(A,u);break;case 6:O=t.hb<t.ib?Mt(t.hb):Mt(t.ib),ut=t.kb<t.lb?Mt(t.kb):Mt(t.lb),K=Mt(d.Math.abs(t.ib-t.hb)),Q=Mt(d.Math.abs(t.lb-t.kb)),R.F3(A,(E=(v=R.B3(A),v!=0?v:kr),Lt.w4(E,(y=dr,h.akb(w.SAb(y).substr(0,w.SAb(On).length),On)||h.akb(w.SAb(y).substr(0,w.SAb(xn).length),xn)||h.akb(w.SAb(y).substr(0,w.SAb(Nn).length),Nn)?RD:CD)))),R.r3(A,O,ut,K,Q),R.F3(A,u);break;case 8:tt=Mt(t.hb),Dt=Mt(t.kb),t.t==-1||t.t==t.e?(Qt=Mt(t.ib),ot=Mt(t.lb)):(Qt=Mt(n.Ql(t.V,t.t)),ot=Mt(n.Rl(t.V,t.t))),R.F3(A,(r=R.w3(A),Lt.y4(-8388608,r))),R.p3(A,tt,Dt,Qt,ot),R.F3(A,u);}},C.hM=function(t){var A;return !!(t.r&&(A=R.M2(t.r),C.iL(t,A)));},C.iM=function(t,A){return n.Rm(t.V,A)&&(C.ML(t,A)!=-1||C.NL(t,A)!=-1);},C.jM=function(t,A,e){var r,P;for(P=false,r=0;r<t.V.f;r++)n.Ll(t.V,r)!=0&&(e&&!n.Am(t.V,r)||A&&n.Am(t.V,r))&&(P||(C.zM(t),P=true),n.rn(t.V,r,0,false));P&&(A||C.nL(t),C.EM(t,d.Math.max(t.eb,1)));},C.kM=function(t){t.cb&&(n.ol(t.cb,t.V),t.I=t.bb?new n.gk(t.bb):null);},C.lM=function(t){var A,e,r,P;for(P=false,e=0;e<t.V.q;e++)if(n.Pm(t.V,e)){P=true;break;}if(r=C.pL(t,P),r){for(C.zM(t),A=0;A<t.V.q;A++)(!P||n.Pm(t.V,A))&&(n.zn(t.V,A,2*r.a-n.Ql(t.V,A)),n.An(t.V,A,2*r.b-n.Rl(t.V,A)));t.eb=d.Math.max(t.eb,1),W.r1(t.k);}},C.mM=function(t,A){t.r=A;},C.nM=function(t,A,e,r,P,i){t.F=A,t.B=e,t.D=r,t.C=P,t.A=i;},C.oM=function(t,A){t.H=A,t.eb=d.Math.max(t.eb,1),W.r1(t.k);},C.pM=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c,b;if(u=-1,P=C.ML(t,A),D=P==-1?C.NL(t,A):-1,e!=0){for(c=-1,L=0;L<t.V.f;L++)L!=P&&n.Hl(t.V,L)==e&&(!n.Qm(t.V,A)||!n.Pm(t.V,L))&&(Q=n.Gl(t.V,L),c<Q&&(c=Q));for(y=0;y<t.V.g;y++)y!=D&&n.cm(t.V,y)==e&&(!n.Qm(t.V,A)||!n.Qm(t.V,y))&&(Q=n.bm(t.V,y),c<Q&&(c=Q));if((P==-1?n.cm(t.V,D):n.Hl(t.V,P))!=e)u=d.Math.min(c+1,31);else if(u=P==-1?n.bm(t.V,D):n.Gl(t.V,P),n.Qm(t.V,A)){for(b=true,T=0;T<t.V.f;T++)if(T!=P&&n.Pm(t.V,T)&&n.Hl(t.V,T)==e&&n.Gl(t.V,T)!=u){b=false;break;}for(l=0;l<t.V.g;l++)if(l!=D&&n.Qm(t.V,l)&&n.cm(t.V,l)==e&&n.bm(t.V,l)!=u){b=false;break;}b&&(u<=c?(++u,u==32&&(u=0)):u=0);}else u<=c?(++u,u==32&&(u=0)):u=0;}if(n.Qm(t.V,A))for(l=0;l<t.V.g;l++)n.Qm(t.V,l)&&n.Rm(t.V,l)&&(r=C.ML(t,l),i=C.NL(t,l),r!=-1?n.on(t.V,r,e,u):i!=-1&&n.Gn(t.V,i,e,u));else P!=-1?n.on(t.V,P,e,u):D!=-1&&n.Gn(t.V,D,e,u);},C.qM=function(t,A){t.V!=A&&(C.zM(t),t.V=A,t.U=0,t.I=null,t.eb=d.Math.max(t.eb,4),W.r1(t.k),C.JL(t,new C.fL(t,1,false)));},C.rM=function(t,A){var e,r,P,i;for(n._v(t.V),t.L=B(o.Q8,GA,26,A.g.a.length+A.f.a.length,0,1),t.Y=A.g.a.length,P=0;P<A.g.a.length+A.f.a.length;P++)t.L[P]=P<A.g.a.length?U.xj(A.g,P):U.xj(A.f,P-A.g.a.length),n.Yk(t.V,t.L[P]);for(n.On(t.V,A.c||At.gH(A)),C.zM(t),t.M=B(o.Q7,H,6,t.V.q,15,1),e=0,r=0;r<t.L.length;r++)for(i=0;i<t.L[r].q;i++)t.M[e++]=r;t.I=new n.fk(),t.U=5,t.eb=d.Math.max(t.eb,5),W.r1(t.k),C.JL(t,new C.fL(t,1,false));},C.sM=function(t,A){t.db=A;},C.uM=function(t,A,e){var r;return r=U.xj(t.I,0),At.EH(r,n.Ql(t.V,A),n.Rl(t.V,A))==At.EH(r,n.Ql(t.V,e),n.Rl(t.V,e));},C.vM=function(t,A){var e,r;t.b&&(C.zM(t),r=(t.U&4)!=0,e=new C.BK(t._,t.V,A,r),C.AK(e,new C.KM(t)));},C.wM=function(t,A){var e;t.b&&(C.zM(t),e=new C.PK(t._,t.V,A),C.OK(e,new C.MM(t)));},C.xM=function(t,A,e){var r;C.zM(t),r=A==-1?new C.$K(t._,t,t.F,t.B,t.D,t.C,t.A):new C._K(t._,t,t.V,A),C.ZK(r,new C.OM(t,A,e));},C.yM=function(t,A,e){var r,P,i,D,u,Q,l,L,y,T,c;for(y=U1(o.Q7,[E1,H],[7,6],15,[e,(t.U&6)!=0?2:1],2),Q=0;Q<e;Q++)y[Q][0]=Q;if(L=C.qL(t,A,e),(t.U&4)!=0)for(t.Y=0,r=(t.U&4)!=0?U.xj(t.I,0):null,l=0;l<e;l++)y[l][1]=At.EH(r,L[l].a,L[l].b)?1:0,y[l][1]==0&&++t.Y;else if((t.U&2)!=0)for(t.Y=e,P=0;P<t.V.q;P++)n.Tl(t.V,P)==0&&y[A[P]][1]==0&&(y[A[P]][1]=1,--t.Y);for(D=L,U.Vrb(y,new C.XM(t,D)),T=B(o.Q7,H,6,e,15,1),u=0;u<e;u++)c=y[u][0],T[c]=u;for(i=0;i<t.V.q;i++)A[i]=T[A[i]];},C.zM=function(t){!t.cb&&(t.cb=new n.eo()),n.ol(t.V,t.cb),t.bb=t.I?new n.gk(t.I):null;},C.AM=function(t){var A,e,r,P,i,D,u,Q,l,L,y,T,c,b,g,k,v,S,p,E;if(p=n.no(t.hb,t.kb,t.ib,t.lb),v=t.ib-t.hb,S=t.lb-t.kb,y=t.q,T=0,b=0,c=0,g=0,y>0&&(T=Jt(b1(d.Math.round(t.o[0]))),b=Jt(b1(d.Math.round(t.p[0])))),y>1&&(c=Jt(b1(d.Math.round(t.o[1]))),g=Jt(b1(d.Math.round(t.p[1])))),Q=0,t.e==-1||n.Mo(t.V,t.e)==0)Q=c0*nn(b1(d.Math.round(p*3/A1)));else if(n.Mo(t.V,t.e)==1)e=n.Yl(t.V,n.Zo(t.V,t.e,0),t.e),r=e-c0,P=e+c0,Q=d.Math.abs(n.oo(p,r))<d.Math.abs(n.oo(p,P))?r:P;else {for(i=B(o.O7,Ft,6,n.Mo(t.V,t.e),15,1),L=0;L<n.Mo(t.V,t.e);L++)i[L]=n.Yl(t.V,t.e,n.Zo(t.V,t.e,L));for(w.lAb(i,w.SAb(T0(U.Xrb.prototype.ld,U.Xrb,[]))),l=0;l<i.length;l++){if(k=l==0?i[i.length-1]-N1:i[l-1],k<p&&p<i[l]){Q=(i[l]+k)/2;break;}if(k<p-N1&&p-N1<i[l]){Q=(i[l]+k)/2;break;}}}if(A=n.Vl(t.V,24*t.ab),t.q=d.Math.abs(n.oo(p,Q))>c0?0:Mt(d.Math.sqrt(v*v+S*S)/A),t.q>0){for((t.o==null||t.o.length<t.q)&&(t.o=B(o.O7,Ft,6,t.q,15,1),t.p=B(o.O7,Ft,6,t.q,15,1)),D=B(o.O7,Ft,6,2,15,1),u=B(o.O7,Ft,6,2,15,1),E=n.oo(p,Q)<0?Q-c0:Q+c0,D[0]=A*d.Math.sin(Q),u[0]=A*d.Math.cos(Q),D[1]=A*d.Math.sin(E),u[1]=A*d.Math.cos(E),L=0;L<t.q;L++)t.o[L]=(L==0?t.hb:t.o[L-1])+D[L&1],t.p[L]=(L==0?t.kb:t.p[L-1])+u[L&1];for(t.n=B(o.Q7,H,6,t.q,15,1),l=0;l<t.q;l++)t.n[l]=n.xl(t.V,t.o[l],t.p[l]),t.n[l]!=-1&&(t.o[l]=n.Ql(t.V,t.n[l]),t.p[l]=n.Rl(t.V,t.n[l]));}return y!=t.q||t.q!=0&&(T!=Jt(b1(d.Math.round(t.o[0])))||b!=Jt(b1(d.Math.round(t.p[0]))))||t.q>1&&(c!=Jt(b1(d.Math.round(t.o[1])))||g!=Jt(b1(d.Math.round(t.p[1]))));},C.BM=function(t,A){var e,r,P,i,D,u,Q,l,L,y,T;if(y=In,A!=-1){for(e=B(o.O7,Ft,6,9,15,1),D=0;D<n.Mo(t.V,A);D++)e[D]=n.Yl(t.V,A,n.Zo(t.V,A,D));if(n.Mo(t.V,A)==1)e[0]<-2.6179938779914944?y=c0:e[0]<ue?y=In:e[0]<qP?y=c0:e[0]<0?y=In:e[0]<Sn?y=PP:e[0]<v0?y=be:e[0]<2.6179938779914944?y=PP:y=be;else {for(u=n.Mo(t.V,A)-1;u>0;u--)for(Q=0;Q<u;Q++)e[Q]>e[Q+1]&&(T=e[Q],e[Q]=e[Q+1],e[Q+1]=T);for(e[n.Mo(t.V,A)]=e[0]+N1,L=0,l=0,i=0;i<n.Mo(t.V,A);i++)r=e[i+1]-e[i],l<r&&(l=r,L=i);y=(e[L]+e[L+1])/2;}}P=n.Vl(t.V,24*t.ab),t.ib=(A==-1?t.hb:n.Ql(t.V,A))+P*d.Math.sin(y),t.lb=(A==-1?t.kb:n.Rl(t.V,A))+P*d.Math.cos(y);},C.CM=function(t,A){t.w!=A&&(t.w==19||A==19?(t.w=A,t.eb=d.Math.max(t.eb,1),W.r1(t.k)):t.w=A);},C.DM=function(t,A,e,r){var P,i,D,u,Q,l;if(D=n.xl(t.V,A,e),u=-1,r&&t.X==8&&D!=-1&&(!C.lL(t,t.e,D)||n.Ll(t.V,t.e)!=0&&n.Ll(t.V,t.e)==n.Ll(t.V,D)&&!n.Am(t.V,t.e)||C.uM(t,t.e,D))&&(D=-1),D!=-1&&(t.w==101||t.w==103||t.w==102?(u=n.Ip(t.V,D),D=-1):t.w==19&&D>=t.V.f&&(D=-1)),u==-1&&D==-1&&t.w!=22&&t.w!=19&&t.w!=3&&t.w!=10&&t.w!=27&&t.w!=16&&t.w!=11&&t.w!=12&&t.w!=13&&t.w!=28&&t.w!=29&&t.w!=30&&t.w!=14&&t.w!=31&&t.w!=15&&t.w!=32&&t.w!=33&&(u=n.yl(t.V,A,e)),u!=-1&&(t.w==101||t.w==103||t.w==102)&&!(n.Rm(t.V,u)&&(C.ML(t,u)!=-1||C.NL(t,u)!=-1))&&(u=-1),P=t.v,!r&&t.I&&(P=null,D==-1&&u==-1&&(t.w==2||t.w==4||t.w==21))){for(l=new U.rrb(t.I);l.a<l.c.a.length;)if(Q=U.qrb(l),(t.w==2||t.w==4&&!Zt(Q,61)||t.w==21&&Zt(Q,94))&&Q.Eb(A,e)){P=Q,t.v&&t.v!=Q&&t.v.Fb();break;}}return i=t.t!=D||t.u!=u||t.v!=P||!!P,t.t!=D&&(t.t!=-1&&w.SAb(t.i.a).length!=0?C.GL(t,t.i.a):(t.t=D,h.whb(t.i,0),C.JL(t,new C.fL(t,4,true)))),t.u!=u&&(t.u=u,C.JL(t,new C.fL(t,8,true))),t.v=P,i;},C.EM=function(t,A){t.eb=d.Math.max(t.eb,A),W.r1(t.k),C.JL(t,new C.fL(t,1,true));},C.FM=function(t){var A;switch(A=-1,t.w){case 18:A=9;break;case 2:t.t!=-1&&n.Pm(t.V,t.t)||t.u!=-1&&n.Qm(t.V,t.u)?A=t.W?4:t.Z?3:2:t.t!=-1||t.u!=-1?A=11:t.v?A=t.W?4:t.Z&&!Zt(t.v,61)?3:2:A=t.Z?t.d?8:6:t.d?7:5;break;case 4:A=1;break;case 22:A=0;break;case 21:A=12;break;default:A=11;}t.s!=A&&(t.s=A,W.i2(t._,A));},C.GM=function(t,A){C.zM(t),C.rM(t,At.UH(A,true,null));},C.HM=function(t,A,e,r){this.V=t,this.U=A,this._=e,this.k=r,this.S=new U.Mj(),this.w=5,this.t=-1,this.u=-1,this.v=null,this.e=-1,this.F=6,this.B=0,this.D=-1,this.C=0,this.A=null,this.b=true,this.a=false,this.X=0,this.s=11,this.i=new h.Kkb(),this.c=331,this.$=1,this.ab=(m1.sO==-1&&(m1.sO=m1.uO()),m1.sO),this.T=Mt(d.Math.round((m1.sO==-1&&(m1.sO=m1.uO()),m1.sO*24))),(this.U&6)!=0&&(this.U|=1),(this.U&12)!=0&&(this.I=new n.fk()),this.eb=4;},x(286,1,Tn,C.HM),s.mc=function(t){var A;Zt(t,140)?C.DL((A=this,A)):Zt(t,156)?C.EL(this,t):Zt(t,137)&&C.FL(this,t);},s.a=false,s.b=false,s.c=0,s.d=false,s.e=0,s.f=0,s.g=false,s.q=0,s.s=0,s.t=0,s.u=0,s.w=0,s.B=0,s.C=0,s.D=0,s.F=0,s.H=0,s.J=0,s.K=0,s.N=0,s.O=false,s.T=0,s.U=0,s.W=false,s.X=0,s.Y=0,s.Z=false,s.$=0,s.ab=0,s.eb=0,s.fb=0,s.hb=0,s.ib=0,s.kb=0,s.lb=0,C.gL=true,o.Mab=N(286),C.JM=function(t,A){C.XL(t.a,A);},C.KM=function(t){this.a=t;},x(293,1,{},C.KM),o.Eab=N(293),C.LM=function(t,A){C.YL(t.a,A);},C.MM=function(t){this.a=t;},x(295,1,{},C.MM),o.Fab=N(295),C.NM=function(t,A){C.ZL(t.a,t.b,t.c,A);},C.OM=function(t,A,e){this.a=t,this.b=A,this.c=e;},x(296,1,{},C.OM),s.b=0,o.Gab=N(296),C.PM=function(){},x(297,1,{},C.PM),s.nc=function(){},o.Hab=N(297),C.RM=function(t){C.$L(t.a);},C.SM=function(t){this.a=t;},x(298,1,{},C.SM),s.nc=function(){C.RM(this);},o.Iab=N(298),C.UM=function(t,A,e){this.a=t,this.c=A,this.b=e;},x(301,1,{},C.UM),o.Jab=N(301),C.VM=function(t,A,e){this.a=t,this.c=A,this.b=e;},x(302,1,{262:1},C.VM),s.oc=function(){C.aM(this.a,this.c,this.b);},s.c=0,o.Kab=N(302),C.XM=function(t,A){this.a=t,this.b=A;},x(303,1,{},C.XM),s.Rb=function(t,A){return C.bM(this.a,this.b,t,A);},s.mb=function(t){return this===t;},o.Lab=N(303),C.aN=function(){C.aN=f1,C.$M=Mt(d.Math.round(Mt(d.Math.round((m1.sO==-1&&(m1.sO=m1.uO()),m1.sO*2))))),C._M=(m1.sO==-1&&(m1.sO=m1.uO()),m1.sO*21);},C.bN=function(t,A,e,r,P){var i;r|=r1,i=C.fN(e),r!=-1&&(R.F3(A,r),R.v3(A,i[0],i[1],C._M,C._M)),e==20?R.o3(A,t.c,0,t.d*84,84,84,i[0],i[1],C._M,C._M):R.o3(A,P?t.g:t.i,(i[0]-C.$M)*t.j,(i[1]-C.$M)*t.j,84,84,i[0],i[1],C._M,C._M);},C.cN=function(t,A){Zt(A,137)?C.eN(t,A):Zt(A,156)&&C.dN(t,A);},C.dN=function(t,A){var e;e=A.b,A.j==1&&(e==-9&&t.b>=17?C.kN(t,t.b-17):e==-10&&t.b<17?C.kN(t,t.b+17):e==-11&&t.b%17>0?C.kN(t,t.b-1):e==-12&&t.b%17<16?C.kN(t,t.b+1):e==122?C.kN(t,18):e==32?C.kN(t,2):e==48?C.kN(t,4):e==116?C.kN(t,21):e==49?C.kN(t,5):e==50?C.kN(t,22):e==109?C.kN(t,19):e==51?C.kN(t,7):e==52?C.kN(t,24):e==53?C.kN(t,8):e==54?C.kN(t,25):e==55?C.kN(t,9):e==97?C.kN(t,26):e==45?C.kN(t,27):e==43?C.kN(t,10):e==117?C.kN(t,6):e==100?C.kN(t,23):e==99?C.kN(t,11):e==110?C.kN(t,12):e==112?C.kN(t,29):e==111?C.kN(t,13):e==115?C.kN(t,30):e==102?C.kN(t,14):e==108?C.kN(t,31):e==98?C.kN(t,15):e==105?C.kN(t,32):e==104?C.kN(t,16):e==46&&C.kN(t,33));},C.eN=function(t,A){var e,r;if(A.j==1){if(e=C.gN(t,A),!(e>=0&&e<34&&(e!=t.b||e==20||e==33)&&(e!=19||(t.a.U&4)!=0)&&(e!=21||(t.a.U&8)!=0)))return;e==20&&e==t.b&&(t.d=++t.d%3),t.k=e,W.R1(t.n);}else if(A.j==2){if(t.k==-1)return;if(r=C.gN(t,A),r!=t.k||t.k==19&&(t.a.U&4)==0||t.k==21&&(t.a.U&8)==0){t.k=-1,W.R1(t.n);return;}if(t.k=-1,r==0||r==1||r==17){W.R1(t.n),C.oL(t.a,r);return;}t.b=r,W.R1(t.n),t.b==20?C.CM(t.a,t.d==0?101:t.d==1?103:102):t.b==33?C.xM(t.a,-1,new C.nN(t,r)):C.CM(t.a,r);}else (A.j==6||A.j==5)&&(e=C.gN(t,A),e==t.k&&(e=-1),e!=t.f&&(t.f=e,W.R1(t.n)));},C.fN=function(t){var A;return A=B(o.O7,Ft,6,2,15,1),A[0]=C._M*(t/17|0)+C.$M,A[1]=C._M*(t%17)+C.$M,A;},C.gN=function(t,A){var e,r,P;return r=A.g-C.$M,P=A.i-C.$M,r<0||r>=2*C._M||P<0||P>=17*C._M?-1:(e=17*Mt(r/C._M)+Mt(P/C._M),e>=0&&e<34&&(e!=t.b||e==20||e==33)&&(e!=19||(t.a.U&4)!=0)&&(e!=21||(t.a.U&8)!=0)?e:-1);},C.hN=function(t){t.i=W.b2(t.a._,JP),m1.zO(t.i),t.g=W.b2(t.a._,JP),m1.tO(t.g),t.o=2*C.$M+m1.vO(R.L3(t.i)/4|0),t.e=2*C.$M+m1.vO(R.I3(t.i)/4|0),t.c=W.b2(t.a._,SD),t.j=(m1.sO==-1&&(m1.sO=m1.uO()),4/m1.sO),t.b=5,t.k=-1,t.f=-1,C.sM(t.a,t);},C.iN=function(t,A){C.CM(t.a,A);},C.jN=function(t,A){var e,r,P,i,D,u,Q;e=W.N1(t.n),P=(Lt.t4(),(Lt.s4[0]*((e&W0)>>16)+Lt.s4[1]*((e&Y0)>>8)+Lt.s4[2]*(e&255))/255<.5),r=P?Lt.u4(e,Dn):e&G0|Mt(d.Math.round(ye*((e&W0)>>16)))<<16|Mt(d.Math.round(ye*((e&Y0)>>8)))<<8|Mt(d.Math.round(ye*(e&255))),D=P?Lt.u4(e,.5):e&G0|Mt(d.Math.round(hn*((e&W0)>>16)))<<16|Mt(d.Math.round(hn*((e&Y0)>>8)))<<8|Mt(d.Math.round(hn*(e&255))),Q=R.L3(t.i),u=R.I3(t.i),R.o3(A,t.i,0,0,Q,u,C.$M,C.$M,Q/t.j,u/t.j),i=C.fN(20),R.o3(A,t.c,0,t.d*84,84,84,i[0],i[1],C._M,C._M),(t.a.U&4)==0&&C.bN(t,A,19,-1,true),(t.a.U&8)==0&&C.bN(t,A,21,-1,true),C.bN(t,A,t.b,D,false),t.f!=-1&&t.f!=t.k&&C.bN(t,A,t.f,r,false),t.k!=-1&&C.bN(t,A,t.k,7167924,false);},C.kN=function(t,A){t.b!=A&&A>=0&&A<34&&(A!=t.b||A==20||A==33)&&(A!=19||(t.a.U&4)!=0)&&(A!=21||(t.a.U&8)!=0)&&!(A==0||A==17||A==1)&&(t.b=A,C.CM(t.a,A),W.R1(t.n));},C.lN=function(t,A){C.aN(),this.n=t,this.a=A,C.hN(this);},x(304,1,Tn,C.lN),s.mc=function(t){C.cN(this,t);},s.b=0,s.d=0,s.e=0,s.f=0,s.j=0,s.k=0,s.o=0,C.$M=0,C._M=0,o.Oab=N(304),C.nN=function(t,A){this.a=t,this.b=A;},x(306,1,{},C.nN),s.nc=function(){C.iN(this.a,this.b);},s.b=0,o.Nab=N(306),C.sN=function(){C.sN=f1,C.qN=j(F(o.Q7,1),H,6,15,[0,2,1,3]),C.rN=j(F(o.Ocb,1),yt,2,6,["plain","italic","bold","italics & bold"]),C.pN=j(F(o.Ocb,1),yt,2,6,["8","9","10","12","14","18","24","32"]);},C.tN=function(t){var A,e,r,P,i,D,u,Q,l,L,y;for(t.b=R._2(t.c),i=C.pN,u=0,l=i.length;u<l;++u)r=i[u],R.Q2(t.b,r);for(R.U2(t.b,true),R.W2(t.b,""+Mt(t.e.c)),t.a=R._2(t.c),P=C.rN,D=0,Q=P.length;D<Q;++D)r=P[D],R.Q2(t.a,r);for(L=0,e=0;e<C.qN.length;e++)if(t.e.d==C.qN[e]){L=e;break;}R.V2(t.a,L),A=j(F(o.Q7,1),H,6,15,[8,-2,4,-2,8]),y=j(F(o.Q7,1),H,6,15,[8,-2,4,-2,8,-2,8]),R.e3(t.c,A,y),R.Y2(t.c,R.a3(t.c,"Text size:"),1,1),R.Y2(t.c,t.b,3,1),R.Y2(t.c,R.a3(t.c,"Text style:"),1,3),R.Y2(t.c,t.a,3,3),t.d=R.b3(t.c,20,3),R.Z3(t.d,t.e.e),R.Z2(t.c,t.d,1,5,3,5);},C.uN=function(t,A){var e,r;if(A.j==0){try{e=h.Mib(R.S2(t.b),r1,qt);}catch(P){if(P=C1(P),Zt(P,40)){R.g3(t.c,"Illegal text size.");return;}else throw st(P);}r=C.qN[R.R2(t.a)],n.Zw(t.e,R.Y3(t.d),e,r);}R.c3(t.c);},C.vN=function(t){C._L(t.a,t.c,t.b);},C.wN=function(t,A){R.f3(t.c,new C.zN(A));},C.xN=function(t,A){C.sN(),this.c=W._1(t,"Edit Text",this),this.e=A,C.tN(this);},x(300,1,Tn,C.xN),s.mc=function(t){C.uN(this,t);},o.Qab=N(300),C.zN=function(t){this.a=t;},x(406,1,{},C.zN),s.nc=function(){C.vN(this.a);},o.Pab=N(406),ht.BN=function(t,A,e){ht.eL.call(this,A,t),this.a=e;},x(140,162,{140:1},ht.BN),s.a=0,o.Rab=N(140),ht.GN=function(){ht.GN=f1,ht.EN=j(F(o.Q7,2),E1,7,0,[j(F(o.Q7,1),H,6,15,[-1432180054,-1434146134,-1441453406,-1471067516,-1589095916,-2060839854,1420461386,1454022954,1454024874]),j(F(o.Q7,1),H,6,15,[-1431677611,-1431764987,-1432092655,-1433403327,-1438646015,-1459616762,-1543499750,-1879031702,1431634346,1073809066,1073814186,1073834666,1431677610]),j(F(o.Q7,1),H,6,15,[-1431737686,-1453057622,-1542187926,-1542188954,-1459351471,-1459351487,KP,1094713345,1077936134,-1879048186,-1543503866,-1543503846,-1459617766,ZP,Gn,Gn]),j(F(o.Q7,1),H,6,15,[-1431737686,-1453057622,-1542187926,-1542188954,-1459351471,-1459351487,KP,1094713345,1077952518,-1879031802,-1543154682,-1543487462,-1459601382,ZP,Gn,Gn]),j(F(o.Q7,1),H,6,15,[mr,mr,-1431741782,-1437330070,-1459351462,-1459351482,KP,-1858076671,-1874853882,-1879048186,-1543503866,-1543503846,-1459617766,ZP,Gn,Gn]),j(F(o.Q7,1),H,6,15,[mr,-1432005270,-1454374890,-1543503871,-1879048191,1073741825,1073741846,1073743210,-1795073366,1079421610,1147841194,-1783977302,vr,vr,-1498764630]),j(F(o.Q7,1),H,6,15,[mr,-1432005270,-1454374890,-1543438335,-1878982655,1075138561,1073807382,1073808746,-1795073366,1079421610,1147841194,-1783977302,vr,vr,-1498764630]),j(F(o.Q7,1),H,6,15,[1431655786,jA,cn,jA,cn,jA,cn,jA,cn,jA,cn,jA,1145324650]),j(F(o.Q7,1),H,6,15,[1431655786,jA,cn,jA,cn,jA,1073758314,1073758250,1074091114,1073758250,1073758314,jA,1145324650]),j(F(o.Q7,1),H,6,15,[-1432003926,-1437248086,1767548266,1437248090,1437248086,1433053846,1433037477,-1431672155,-1432004955,1433037477,1433037462,1437248086,1437248090,1767548266,-1437248086,-1432003926]),null]),ht.CN=j(F(o.Q7,1),H,6,15,[1,7,7,5,8,7,8,7,8,7,2,14,2,14,0,0,0,0,8,8,0,0]),ht.DN=j(F(o.Q7,1),H,6,15,[4,20,10,29,18,18,18,18,18,18,16,11,16,11,4,5,4,5,16,16,16,16,0,0,0,0,12,2]),ht.FN=j(F(o.Ocb,1),yt,2,6,["chain.png","eraser.png","hand.png","handPlus.png","fist.png","lasso.png","lassoPlus.png","rect.png","rectPlus.png","zoom.png","invisible.png",null,null,"pointingHand.png"]);},ht.HN=function(t,A){ht.GN();var e,r,P,i;if(ht.EN[A]!=null){for(r=0;r<ht.EN[A].length;r++)for(i=ht.EN[A][r],P=15;P>=0;P--)(i&3)==0&&R.M3(t,P,r,-1),i>>=2;for(e=0;e<ht.EN[A].length;e++)for(i=ht.EN[A][e],P=15;P>=0;P--)(i&3)==1&&R.M3(t,P,e,G0),i>>=2;}},ht.IN=function(t){n._c(),n.ce.call(this,t);},x(105,121,{121:1,105:1},ht.IN),s.qb=function(t){R.p3(this.B,t.a,t.c,t.b,t.d);},s.rb=function(t){R.n3(this.B,t.a,t.c,t.b,t.d);},s.sb=function(t){R.u3(this.B,new R.U3(t));},s.tb=function(t,A,e){R.m3(this.B,A,e,t);},s.ub=function(t,A,e){R.t3(this.B,t,A,e);},s.vb=function(t){return R.x3(this.B,t).b;},s.wb=function(){return this.b;},s.xb=function(){n.vd(this),this.a=1;},s.Ab=function(t){this.a=t,R.E3(this.B,this.a);},s.Bb=function(t){R.F3(this.B,t);},s.Cb=function(t){this.b=t,this.B&&R.D3(this.B,t,false,false);},s.a=0,s.b=0,o.Sab=N(105),ht.UN=function(t,A){U.sj(t.a,A);},ht.VN=function(t,A){var e,r;for(r=new U.rrb(t.a);r.a<r.c.a.length;)e=U.qrb(r),e.mc(A);},ht.WN=function(){this.a=new U.Mj();},x(221,1,{}),o.Tab=N(221),ht.XN=function(t,A,e,r){ht.eL.call(this,t,r),this.b=A,this.a=e;},x(156,162,{156:1},ht.XN),s.a=false,s.b=0,o.Vab=N(156),ht.ZN=function(){ht.ZN=f1,ht.YN=j(F(o.Ocb,1),yt,2,6,["none","pressed","released","clicked","entered","exited","moved","dragged"]);},ht.$N=function(t,A,e,r,P,i,D,u,Q,l){ht.ZN(),ht.eL.call(this,t,l),this.b=A,this.c=e,this.g=r,this.i=P,this.f=i,this.d=D,this.a=u,this.e=Q;},x(137,162,{137:1},ht.$N),s.pb=function(){return ht.YN[this.j]+" x:"+this.g+" y:"+this.i+" button:"+this.b+" clicks:"+this.c+(this.f?" shift":"")+(this.d?" ctrl":"")+(this.a?" alt":"")+" isPopup:"+(this.e?"y":"n");},s.a=false,s.b=0,s.c=0,s.d=false,s.e=false,s.f=false,s.g=0,s.i=0,o.Wab=N(137),ht.aO=function(){},ht.bO=function(t,A){this.a=t,this.b=A;},x(55,1,{55:1},ht.aO,ht.bO),s.pb=function(){return "x:"+this.a+" y:"+this.b;},s.a=0,s.b=0,o.Xab=N(55),ht.dO=function(t,A,e){t.a==t.b.length&&(t.b=U.yrb(t.b,2*t.a),t.c=U.yrb(t.c,2*t.a)),t.b[t.a]=A,t.c[t.a]=e,++t.a;},ht.eO=function(t,A){return t.b[A];},ht.fO=function(t,A){return t.c[A];},ht.gO=function(t){t.a>0&&--t.a;},ht.hO=function(){this.b=B(o.O7,Ft,6,64,15,1),this.c=B(o.O7,Ft,6,64,15,1),this.a=0;},ht.iO=function(t){this.b=B(o.O7,Ft,6,t,15,1),this.c=B(o.O7,Ft,6,t,15,1),this.a=0;},x(118,1,{415:1},ht.hO,ht.iO),s.Hb=function(t,A){var e,r,P;for(P=false,r=this.a-1,e=0;e<this.a;e++)this.c[e]>A!=this.c[r]>A&&t<(this.b[r]-this.b[e])*(A-this.c[e])/(this.c[r]-this.c[e])+this.b[e]&&(P=!P),r=e;return P;},s.a=0,o.Yab=N(118),ht.kO=function(t,A,e){return A>=t.c&&A<=t.c+t.b&&e>=t.d&&e<=t.d+t.a;},ht.lO=function(t,A){return ht.kO(t,A.c,A.d)&&ht.kO(t,A.c+A.b,A.d+A.a);},ht.mO=function(t,A,e,r,P){t.c=A,t.d=e,t.b=r,t.a=P;},ht.nO=function(t,A){var e,r,P,i;return P=d.Math.min(t.c,A.c),i=d.Math.min(t.d,A.d),r=d.Math.max(t.c+t.b,A.c+A.b)-P,e=d.Math.max(t.d+t.a,A.d+A.a)-i,new ht.pO(P,i,r,e);},ht.oO=function(){},ht.pO=function(t,A,e,r){this.c=t,this.d=A,this.b=e,this.a=r;},x(35,1,{35:1,415:1},ht.oO,ht.pO),s.Hb=function(t,A){return ht.kO(this,t,A);},s.pb=function(){return "x:"+this.c+" y:"+this.d+" w:"+this.b+" h:"+this.a;},s.a=0,s.b=0,s.c=0,s.d=0,o.Zab=N(35),m1.tO=function(t){var A,e,r,P,i;for(e=Lt.v4(new j1.Bgb(238,238,238)),r=WA&e.a,P=0;P<R.L3(t);P++)for(i=0;i<R.I3(t);i++)A=R.K3(t,P,i),R.M3(t,P,i,(G0&A)+r);},m1.uO=function(){return globalThis.devicePixelRatio||1;},m1.vO=function(t){return Mt(d.Math.round((m1.sO==-1&&(m1.sO=m1.uO()),m1.sO*t)));},m1.sO=-1,m1.yO=function(){m1.yO=f1,m1.xO=j(F(o.Q7,1),H,6,15,[5258420,0]),m1.wO=j(F(o.Q7,1),H,6,15,[11837695,14737632]);},m1.zO=function(t){m1.yO();var A;A=dr,(h.akb(w.SAb(A).substr(0,18),On)||h.akb(w.SAb(A).substr(0,17),xn)||h.akb(w.SAb(A).substr(0,14),Nn))&&m1.AO(t,m1.wO);},m1.AO=function(t,A){var e,r,P,i,D;for(i=0;i<R.L3(t);i++)for(D=0;D<R.I3(t);D++)for(e=R.K3(t,i,D),P=e&WA,r=0;r<m1.xO.length;r++)if(P==m1.xO[r]){R.M3(t,i,D,(G0&e)+A[r]);break;}},m.BO=function(t){t=t||{};var A=0;return t.createSymmetryRank&&(A|=1),t.considerStereoheterotopicity&&(A|=2),t.encodeAtomCustomLabels&&(A|=8),t.encodeAtomSelection&&(A|=16),t.assignParitiesToTetrahedralN&&(A|=32),t.coordsAre3d&&(A|=64),t.createPseudoStereoGroups&&(A|=128),t.distinguishRacemicOrGroups&&(A|=256),t.tieBreakFreeValenceAtoms&&(A|=512),t.encodeAtomCustomLabelsWithoutRanking&&(A|=f0),t.neglectAnyStereoInformation&&(A|=B1),A;},m.CO=function(t,A){this.a=new n.Og(t.a,m.BO(A));},x(412,1,{},m.CO),s.getCanMolecule=function(t){var A;return A=n.kg(this.a,t),new m.cU(A);},s.getEncodedCoordinates=function(t){return n.ng(this.a,t);},s.getEncodedMapping=function(){return n.og(this.a);},s.getFinalRank=function(){return this.a.f;},s.getGraphAtoms=function(){return n.pg(this.a);},s.getGraphIndexes=function(){return n.qg(this.a);},s.getIDCode=function(){return n.rg(this.a);},s.getSymmetryRank=function(t){return n.vg(this.a,t);},s.getSymmetryRanks=function(){return this.a.g;},s.hasCIPParityDistinctionProblem=function(){return this.a.d;},s.invalidateCoordinates=function(){this.a.s=null;},s.normalizeEnantiomer=function(){return n.Fg(this.a);},s.setParities=function(){n.Jg(this.a);},o._ab=N(412),m.VO=function(){},m.WO=function(t,A){switch(A){default:case 0:return n.Th(t.a,0);case 1:return n.Th(t.a,1);case 2:return n.Th(t.a,2);case 3:return n.Th(t.a,3);case 4:return n.Th(t.a,4);}},x(416,1,{},m.VO),m.QO=2,m.RO=0,m.SO=1,m.TO=4,m.UO=3,o.$ab=N(416),m._O=function(t){m.M$||m.P$(),this.a=new _.OBb(o.Kfb(t));},x(417,1,{},m._O),s.getConformerCount=function(){return this.a.p;},s.getNextConformerAsMolecule=function(t){var A,e;return A=t?t.a:null,e=_.CBb(this.a,A),e?e==A?t:new m.cU(e):null;},s.getOneConformerAsMolecule=function(t){var A;return A=_.EBb(this.a,t.a),A?t:null;},s.getPotentialConformerCount=function(){return _.FBb(this.a);},s.qc=function(t,A,e,r){return _.IBb(this.a,t.a,A,e,r);},s.initializeConformers=function(t,A){A===void 0&&(A={});var e=A.strategy;e===void 0&&(e=3);var r=A.maxTorsionSets;r===void 0&&(r=ur);var P=A.use60degreeSteps;return P===void 0&&(P=false),this.qc(t,e,r,P);},m.XO=4,m.YO=3,m.ZO=1,m.$O=2,o.abb=N(417),m.gP=function(){},m.hP=function(t,A,e,r,P){return et.jG(t,A,e,r,P);},x(418,1,{},m.gP),o.bbb=N(418),m.jP=function(){throw new Error("drug likeness must be assessed first");},m.kP=function(){m.M$||m.P$(),this.a=new et.oG();},x(419,1,{},m.kP),s.assessDruglikeness=function(t){var A;return et.mG(this.a,(A=t.a,!V1.b4&&(V1.b4=new V1.c4()),A));},s.getDetail=function(){var t;return t=this.a.a,!t&&m.jP(),V1.g4(t);},s.getDruglikenessString=function(t){return et.nG(t.a);},m.iP=-999,o.cbb=N(419),m.uP=function(t,A,e){m.M$||m.P$(),m.xP(A),this.a=new Z.AC(t.a,A,new U.pub());},m.xP=function(t){h.akb(t,YP)&&!m.rP?(Z.uC(),Z.BC(YP,new Z.sD("/resources/forcefield/mmff94/outofplane.csv","/resources/forcefield/mmff94/torsion.csv")),m.rP=true):h.akb(t,XP)&&!m.sP?(Z.uC(),Z.BC(XP,new Z.sD(HD,"/resources/forcefield/mmff94/94s/torsion.csv")),m.sP=true):h.akb(t,yr)&&!m.tP&&(Z.uC(),Z.BC(yr,new Z.sD(HD,"/resources/forcefield/mmff94/94s/torsionPlus.csv")),m.tP=true);},x(420,1,{},m.uP),s._minimise=function(t,A,e){return gA.zB(this.a,t,A);},s.getTotalEnergy=function(){return Z.wC(this.a);},s.size=function(){return this.a.b.q;},m.oP=YP,m.pP=XP,m.qP=yr,m.rP=false,m.sP=false,m.tP=false,o.dbb=N(420),m.YT=function(){m.YT=f1,m.XT=new ht.oO(),m.dQ=j(F(o.Ocb,1),yt,2,6,["?","H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P","S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr","Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og",M1,M1,M1,M1,M1,M1,M1,M1,M1,M1,"R4","R5","R6","R7","R8","R9","R10","R11","R12","R13","R14","R15","R16","R1","R2","R3","A","A1","A2","A3",M1,M1,"D","T","X","R","H2","H+","Nnn","HYD","Pol",M1,M1,M1,M1,M1,M1,M1,M1,M1,M1,M1,"Ala","Arg","Asn","Asp","Cys","Gln","Glu","Gly","His","Ile","Leu","Lys","Met","Phe","Pro","Ser","Thr","Trp","Tyr","Val"]),m.WT=j(F(o.ufb,1),Yt,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]),m.QR=j(F(o.M7,2),cA,10,0,[null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[0]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[0]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[2,4,6]),j(F(o.M7,1),Y,6,15,[1,3,5,7]),j(F(o.M7,1),Y,6,15,[0]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[2,4,6]),j(F(o.M7,1),Y,6,15,[1,3,5,7]),j(F(o.M7,1),Y,6,15,[0,2]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,j(F(o.M7,1),Y,6,15,[1,2,3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[2,4,6]),j(F(o.M7,1),Y,6,15,[1,3,5,7]),j(F(o.M7,1),Y,6,15,[0,2,4,6]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[2])]),m.hT=j(F(o.M7,2),cA,10,0,[null,j(F(o.M7,1),Y,6,15,[1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),null,null,j(F(o.M7,1),Y,6,15,[-3]),j(F(o.M7,1),Y,6,15,[-2]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),null,j(F(o.M7,1),Y,6,15,[-3]),j(F(o.M7,1),Y,6,15,[-2]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3,4]),j(F(o.M7,1),Y,6,15,[2,3,4,5]),j(F(o.M7,1),Y,6,15,[2,3,6]),j(F(o.M7,1),Y,6,15,[2,3,4,7]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[1,2]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[-3,3,5]),j(F(o.M7,1),Y,6,15,[-2]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[6]),j(F(o.M7,1),Y,6,15,[4,6,7]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[-3,3,5]),j(F(o.M7,1),Y,6,15,[-2,4,6]),j(F(o.M7,1),Y,6,15,[-1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[5]),j(F(o.M7,1),Y,6,15,[6]),j(F(o.M7,1),Y,6,15,[4,6,7]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[1,3]),j(F(o.M7,1),Y,6,15,[1,2]),j(F(o.M7,1),Y,6,15,[1,3]),j(F(o.M7,1),Y,6,15,[2,4]),j(F(o.M7,1),Y,6,15,[3,5]),j(F(o.M7,1),Y,6,15,[-2,2,4]),j(F(o.M7,1),Y,6,15,[-1,1]),null,j(F(o.M7,1),Y,6,15,[1]),j(F(o.M7,1),Y,6,15,[2]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[4]),j(F(o.M7,1),Y,6,15,[4,5]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3,4,5,6]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3,4]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[2,3]),j(F(o.M7,1),Y,6,15,[3])]),m.PP=j(F(o.Ocb,1),yt,2,6,[hP,cP,bP]);},m.ZT=function(t,A,e,r){return {x:t,y:A,width:e,height:r};},m.$T=function(t){return t.a;},m._T=function(t,A,e,r){var P;return P=0,A&&(P|=1),e&&(P|=2),r&&(P|=4),n.qr(new n.yr(t.a,P));},m.aU=function(){m.YT(),m.bU.call(this,32,32,null);},m.bU=function(t,A,e){m.YT(),e?this.a=e:this.a=new n.uw(t,A);},m.cU=function(t){m.YT(),m.bU.call(this,0,0,t);},m.GU=function(t,A){return {molecule:t,map:A};},m.WU=function(t,A){m.YT();var e;return typeof A===zn&&(A=true),typeof A===Q0?e=m.YU(t,A):typeof A===H0&&(e=m.XU(t,A)),e;},m.XU=function(t,A){return new m.cU(n.Qq(new n._q(false),t,A));},m.YU=function(t,A){return new m.cU(n.Pq(new n._q(A),t));},m.ZU=function(t){return m.YT(),new m.cU(n.fs(new n.Cs(),t));},m.$U=function(t){m.YT();var A,e,r;return r=new n.Ds(),e=(r.j=null,n.As(r,new Rt.ehb(new Rt.phb(t)))?r.j:null),A=r.f==null?n.lp(r.j):r.f,m.GU(new m.cU(e),A);},m._U=function(t,A){m.YT(),A=A||{};var e=!A.noCoordinates,r=!A.noStereo;return m.aV(t,e,r);},m.aV=function(t,A,e){var r,P,i;return i=new n.Iv(),P=new m.aU(),n.yv(i,P.a,w.sAb((r=t,w.nAb(),r)),A,e),P;},m.iV=function(t,A,e,r){return m.YT(),n.no(t,A,e,r);},m.jV=function(t,A){return m.YT(),n.oo(t,A);},m.JV=function(t,A){return m.YT(),n.po(t,A);},m.mW=function(){return m.YT(),n.Rk(),n.Rk(),n.Qk;},m.sX=function(t){return m.YT(),n.qo(t);},m.tX=function(t){return m.YT(),n.ro(t);},m.RY=function(t){m.YT(),n.Rk(),n.Qk=t;},x(42,1,{42:1},m.aU,m.bU,m.cU),s.addAtom=function(t){return n.Uk(this.a,t);},s.addBond=function(t,A){return n.Vk(this.a,t,A);},s.addFragment=function(t,A,e){n.xo(this.a,t.a,A,e);},s.rc=function(){Xt.Lz(this.a);},s.sc=function(t){Xt.Mz(this.a,t);},s.addImplicitHydrogens=function(t){t===void 0?this.rc():this.sc(t);},s.tc=function(){Xt.sz(),Xt.tz(this.a,1);},s.uc=function(t){Xt.tz(this.a,t);},s.addMissingChirality=function(t){t===void 0?this.tc():this.uc(t);},s.addMolecule=function(t){return n.Yk(this.a,t.a);},s.addOrChangeAtom=function(t,A,e,r,P,i,D){return n.$k(this.a,t,A,e,r,P,i,D);},s.addOrChangeBond=function(t,A,e){return n._k(this.a,t,A,e);},s.addRing=function(t,A,e,r,P){return n.al(this.a,t,A,e,r,P);},s.addRingToAtom=function(t,A,e,r){return n.bl(this.a,t,A,e,r);},s.addRingToBond=function(t,A,e,r){return n.cl(this.a,t,A,e,r);},s.addSubstituent=function(t,A){return n.dl(this.a,t.a,A);},s.calculateTorsion=function(t){return n.el(this.a,t);},s.canonizeCharge=function(t){return n.zo(this.a,t,false);},s.changeAtom=function(t,A,e,r,P){return n.fl(this.a,t,A,e,r,P);},s.changeAtomCharge=function(t,A){return n.hl(this.a,t,A);},s.changeBond=function(t,A){return n.il(this.a,t,A);},s.clear=function(){n._v(this.a);},s.convertStereoBondsToSingleBonds=function(t){n.Bo(this.a,t);},s.copyAtom=function(t,A,e,r){return n.ll(this.a,t.a,A,e,r);},s.copyBond=function(t,A,e,r,P,i){return n.nl(this.a,t.a,A,e,r,P,i);},s.copyMolecule=function(t){n.ol(this.a,t.a);},s.copyMoleculeByAtoms=function(t,A,e,r){n.Co(this.a,t.a,A,e,r);},s.copyMoleculeByBonds=function(t,A,e,r){return n.Do(this.a,t.a,A,e,r);},s.copyMoleculeProperties=function(t){n.aw(this.a,t.a);},s.deleteAtom=function(t){n.ql(this.a,t);},s.deleteAtomOrBond=function(t,A){return n.rl(this.a,t,A);},s.deleteAtoms=function(t){return n.sl(this.a,t);},s.deleteBond=function(t){n.tl(this.a,t);},s.deleteBondAndSurrounding=function(t){n.ul(this.a,t);},s.deleteMarkedAtomsAndBonds=function(){return n.vl(this.a);},s.deleteMolecule=function(){n._v(this.a);},s.deleteSelectedAtoms=function(){return n.wl(this.a);},s.ensureHelperArrays=function(t){n.bw(this.a,t);},s.findAlleneCenterAtom=function(t){return n.Fo(this.a,t);},s.findAlleneEndAtom=function(t,A){return n.Go(this.a,t,A);},s.findAtom=function(t,A){return n.xl(this.a,t,A);},s.findBINAPChiralityBond=function(t){return n.Ho(this.a,t);},s.findBond=function(t,A){return n.yl(this.a,t,A);},s.findRingSystem=function(t,A,e,r){n.Io(this.a,t,A,e,r);},s.getAbsoluteAtomParity=function(t){return n.cw(this.a,t);},s.getAbsoluteBondParity=function(t){return n.dw(this.a,t);},s.getAllAtoms=function(){return this.a.q;},s.getAllBonds=function(){return this.a.r;},s.getAllConnAtoms=function(t){return n.Lo(this.a,t);},s.getAllConnAtomsPlusMetalBonds=function(t){return n.Mo(this.a,t);},s.getAllHydrogens=function(t){return n.No(this.a,t);},s.getAromaticRingCount=function(){return n.Oo(this.a);},s.getAtomAbnormalValence=function(t){return n.zl(this.a,t);},s.getAtomCIPParity=function(t){return n.Al(this.a,t);},s.getAtomCharge=function(t){return n.Bl(this.a,t);},s.getAtomColor=function(t){return n.Cl(this.a,t);},s.getAtomCustomLabel=function(t){return n.El(this.a,t);},s.getAtomESRGroup=function(t){return n.Gl(this.a,t);},s.getAtomESRType=function(t){return n.Hl(this.a,t);},s.getAtomLabel=function(t){return n.Il(this.a,t);},s.getAtomList=function(t){return n.Jl(this.a,t);},s.getAtomListString=function(t){return n.Kl(this.a,t);},s.getAtomMapNo=function(t){return n.Ll(this.a,t);},s.getAtomMass=function(t){return n.Ml(this.a,t);},s.getAtomParity=function(t){return n.Nl(this.a,t);},s.getAtomPi=function(t){return n.Qo(this.a,t);},s.getAtomPreferredStereoBond=function(t){return n.Ro(this.a,t);},s.getAtomQueryFeaturesObject=function(t){return V1._3(this.a,t);},s.getAtomRadical=function(t){return n.Pl(this.a,t);},s.getAtomRingBondCount=function(t){return n.So(this.a,t);},s.getAtomRingCount=function(t,A){return n.To(this.a,t,A);},s.getAtomRingSize=function(t){return n.Uo(this.a,t);},s.getAtomX=function(t){return n.Ql(this.a,t);},s.getAtomY=function(t){return n.Rl(this.a,t);},s.getAtomZ=function(t){return n.Sl(this.a,t);},s.getAtomicNo=function(t){return n.Tl(this.a,t);},s.getAtoms=function(){return this.a.f;},s.getAverageBondLength=function(t){return n.Vo(this.a,t);},s.getAverageTopologicalAtomDistance=function(){return n.Wo(this.a);},s.getBond=function(t,A){return n.Xo(this.a,t,A);},s.getBondAngle=function(t,A){return n.Yl(this.a,t,A);},s.getBondAtom=function(t,A){return n.Zl(this.a,t,A);},s.getBondBridgeMaxSize=function(t){return n.$l(this.a,t);},s.getBondBridgeMinSize=function(t){return n._l(this.a,t);},s.getBondCIPParity=function(t){return n.am(this.a,t);},s.getBondESRGroup=function(t){return n.bm(this.a,t);},s.getBondESRType=function(t){return n.cm(this.a,t);},s.getBondLength=function(t){return n.dm(this.a,t);},s.getBondOrder=function(t){return n.fm(this.a,t);},s.getBondParity=function(t){return n.gm(this.a,t);},s.getBondPreferredStereoBond=function(t){return n.oq(this.a,t);},s.getBondQueryFeatures=function(t){return n.hm(this.a,t);},s.getBondQueryFeaturesObject=function(t){return V1.a4(this.a,t);},s.getBondRingSize=function(t){return n.Yo(this.a,t);},s.getBondType=function(t){return n.im(this.a,t);},s.getBondTypeSimple=function(t){return n.jm(this.a,t);},s.getBonds=function(){return this.a.g;},s.getBounds=function(){var t;return t=n.km(this.a,m.XT),t?m.ZT(t.c,t.d,t.b,t.a):null;},s.getCanonizedIDCode=function(t){var A;return A=new n.Og(this.a,t),n.rg(A);},s.getChiralText=function(){return n.ew(this.a);},s.getChirality=function(){return this.a.I;},s.getCompactCopy=function(){return new m.cU(n.fw(this.a));},s.getConnAtom=function(t,A){return n.Zo(this.a,t,A);},s.getConnAtoms=function(t){return n.$o(this.a,t);},s.getConnBond=function(t,A){return n._o(this.a,t,A);},s.getConnBondOrder=function(t,A){return n.ap(this.a,t,A);},s.getDefaultMaxValenceUncharged=function(t){return n.lm(this.a,t);},s.getDiastereotopicAtomIDs=function(){return Xt.uz(this.a);},s.getElectronValenceCorrection=function(t,A){return n.mm(this.a,t,A);},s.getExcludedNeighbourCount=function(t){return n.bp(this.a,t);},s.getExplicitHydrogens=function(t){return n.cp(this.a,t);},s.getFinalRanks=function(t){var A;return A=new n.Og(this.a,t),A.f;},s.getFisherProjectionParity=function(t,A,e,r){return n.ep(this.a,t,A,e,r);},s.getFragmentAtoms=function(t,A){return n.fp(this.a,t,A);},s.getFragmentNumbers=function(t,A,e){return n.hp(this.a,t,A,e);},s.getFragments=function(){var t,A,e;for(t=n.hw(this.a),e=B(o.fbb,_P,42,t.length,0,1),A=0;A<t.length;A++)e[A]=new m.cU(t[A]);return e;},s.getFreeValence=function(t){return n.jp(this.a,t);},s.getHandleHydrogenMap=function(){return n.lp(this.a);},s.getHelperArrayStatus=function(){return this.a.T;},s.getHoseCodes=function(t){t=t||{};var A=(typeof t.maxSphereSize===zn?5:t.maxSphereSize)|0,e=(typeof t.type===zn?0:t.type)|0;return Xt.wz(this.a,A,e);},s.getIDCode=function(){var t;return t=n.jw(this.a),t==null&&(t=n.rg(new n.Ng(this.a))),t;},s.getIDCodeAndCoordinates=function(){return {idCode:this.getIDCode(),coordinates:this.getIDCoordinates()};},s.getIDCoordinates=function(){var t;return t=n.kw(this.a),t==null&&(t=n.mg(new n.Ng(this.a))),t;},s.getImplicitHigherValence=function(t,A){return n.mp(this.a,t,A);},s.getImplicitHydrogens=function(t){return n.op(this.a,t);},s.getIndex=function(){return V1.h4(this.a);},s.getLowestFreeValence=function(t){return n.pp(this.a,t);},s.getMaxAtoms=function(){return this.a.M;},s.getMaxBonds=function(){return this.a.N;},s.getMaxValence=function(t){return n.om(this.a,t);},s.getMaxValenceUncharged=function(t){return n.pm(this.a,t);},s.getMetalBondedConnAtoms=function(t){return n.rp(this.a,t);},s.getMolecularFormula=function(){return new m.o0(this.a);},s.getMoleculeColor=function(){return this.a.O;},s.getMolweight=function(){return n.sp(this.a);},s.getName=function(){return this.a.P;},s.getNonHydrogenNeighbourCount=function(t){return n.tp(this.a,t);},s.getNumberOfHydrogens=function(){return Xt.Nz(this.a);},s.getOccupiedValence=function(t){return n.vp(this.a,t);},s.getPath=function(t,A,e,r,P){return n.yp(this.a,t,A,e,r,null,P);},s.getPathBonds=function(t,A,e){n.zp(this.a,t,A,e);},s.getPathLength=function(t,A){return n.Ap(this.a,t,A);},s.getRingSet=function(){return new m.V$(n.Ep(this.a));},s.getRotatableBondCount=function(){return n.Fp(this.a);},s.vc=function(t,A,e,r,P,i,D){var u,Q,l,L,y,T,c,b,g,k,v,S;for(l=true,y=0;y<this.a.q-1;y++)if(n.Ql(this.a,y)!=n.Ql(this.a,y+1)||n.Rl(this.a,y)!=n.Rl(this.a,y+1)){l=false;break;}return T=l?n.fw(this.a):this.a,l&&X.jA(new X.CA(0),T),n.bw(T,95),S=V1.j4(D),L=V1.i4(D),Q=new n.Ju(T,L,i),Q.J=e,n.be(Q,null,new ht.pO(0,0,t,A),S),u=Q.v,n.Td(Q,null),v=n.Hu(Q),r?(b=Jt(b1(d.Math.round(u.b+P*2))),c=Jt(b1(d.Math.round(u.a+P*2))),g=Jt(b1(d.Math.round(u.c-P))),k=Jt(b1(d.Math.round(u.d-P))),h.nkb(v,'width="\\d+px" height="\\d+px" viewBox="0 0 \\d+ \\d+"','width="'+b+'px" height="'+c+'px" viewBox="'+g+" "+k+" "+b+" "+c+'"')):v;},s.getStereoBond=function(t){return n.Ip(this.a,t);},s.getStereoCenterCount=function(){return n.lw(this.a);},s.wc=function(){return m.$T(this);},s.getStereoProblem=function(t){return n.rm(this.a,t);},s.getSubstituent=function(t,A,e,r,P){return n.Kp(this.a,t,A,e,r.a,P);},s.getSubstituentSize=function(t,A){return n.Lp(this.a,t,A);},s.getSymmetryRank=function(t){return n.mw(this.a,t);},s.getZNeighbour=function(t,A){return n.Mp(this.a,t,A);},s.invalidateHelperArrays=function(t){n.um(this.a,t);},s.inventCoordinates=function(){var t;t=new X.BA(),t.g=new U.uwb(0),X.jA(t,this.a),n.yq(this.a);},s.is3D=function(){return n.vm(this.a);},s.isAlkaliMetal=function(t){return n.Qp(this.a,t);},s.isAllylicAtom=function(t){return n.Rp(this.a,t);},s.isAmideTypeBond=function(t){return n.Sp(this.a,t);},s.isAromaticAtom=function(t){return n.Tp(this.a,t);},s.isAromaticBond=function(t){return n.Up(this.a,t);},s.isAtomConfigurationUnknown=function(t){return n.wm(this.a,t);},s.isAtomMarkedForDeletion=function(t){return n.xm(this.a,t);},s.isAtomParityPseudo=function(t){return n.ym(this.a,t);},s.isAtomStereoCenter=function(t){return n.zm(this.a,t);},s.isAutoMappedAtom=function(t){return n.Am(this.a,t);},s.isBINAPChiralityBond=function(t){return n.Vp(this.a,t);},s.isBondBackgroundHilited=function(t){return n.Bm(this.a,t);},s.isBondBridge=function(t){return n.Cm(this.a,t);},s.isBondForegroundHilited=function(t){return n.Dm(this.a,t);},s.isBondMarkedForDeletion=function(t){return n.Em(this.a,t);},s.isBondParityPseudo=function(t){return n.Fm(this.a,t);},s.isBondParityUnknownOrNone=function(t){return n.Gm(this.a,t);},s.isChalcogene=function(t){return n.Xp(this.a,t);},s.isDelocalizedBond=function(t){return n.Yp(this.a,t);},s.isEarthAlkaliMetal=function(t){return n.Zp(this.a,t);},s.isElectronegative=function(t){return n.Hm(this.a,t);},s.isElectropositive=function(t){return n.Im(this.a,t);},s.isFlatNitrogen=function(t){return n.$p(this.a,t,true);},s.isFragment=function(){return this.a.K;},s.isHalogene=function(t){return n._p(this.a,t);},s.isMarkedAtom=function(t){return n.Km(this.a,t);},s.isMetalAtom=function(t){return n.Lm(this.a,t);},s.isNaturalAbundance=function(t){return n.Mm(this.a,t);},s.isNitrogenFamily=function(t){return n.bq(this.a,t);},s.isOrganicAtom=function(t){return n.Nm(this.a,t);},s.isPseudoRotatableBond=function(t){return n.cq(this.a,t);},s.isPurelyOrganic=function(){return n.Om(this.a);},s.isRingAtom=function(t){return n.fq(this.a,t);},s.isRingBond=function(t){return n.gq(this.a,t);},s.isSelectedAtom=function(t){return n.Pm(this.a,t);},s.isSelectedBond=function(t){return n.Qm(this.a,t);},s.isSimpleHydrogen=function(t){return n.hq(this.a,t);},s.isSmallRingAtom=function(t){return n.iq(this.a,t);},s.isSmallRingBond=function(t){return n.jq(this.a,t);},s.isStabilizedAtom=function(t){return n.kq(this.a,t);},s.isStereoBond=function(t){return n.Rm(this.a,t);},s.markAtomForDeletion=function(t){n.Um(this.a,t);},s.markBondForDeletion=function(t){n.Vm(this.a,t);},s.normalizeAmbiguousBonds=function(){return n.lq(this.a);},s.removeAtomColors=function(){n.Ym(this.a);},s.removeAtomCustomLabels=function(){this.a.t=null;},s.removeAtomMarkers=function(){n.Zm(this.a);},s.removeAtomSelection=function(){n.$m(this.a);},s.removeBondHiliting=function(){n._m(this.a);},s.removeExplicitHydrogens=function(){n.qq(this.a,true);},s.removeQueryFeatures=function(){return n.bn(this.a);},s.renumberESRGroups=function(t){return n.cn(this.a,t);},s.scaleCoords=function(t){n.dn(this.a,t);},s.setAllAtoms=function(t){n.en(this.a,t);},s.setAllBonds=function(t){n.fn(this.a,t);},s.setAssignParitiesToNitrogen=function(t){n.nw(this.a,t);},s.setAtomAbnormalValence=function(t,A){n.gn(this.a,t,A);},s.setAtomCIPParity=function(t,A){n.hn(this.a,t,A);},s.setAtomCharge=function(t,A){n.jn(this.a,t,A);},s.setAtomColor=function(t,A){n.kn(this.a,t,A);},s.setAtomConfigurationUnknown=function(t,A){n.ln(this.a,t,A);},s.setAtomCustomLabel=function(t,A){n.mn(this.a,t,A);},s.setAtomESR=function(t,A,e){n.on(this.a,t,A,e);},s.setAtomList